(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     38338,       1437]
NotebookOptionsPosition[     29357,       1140]
NotebookOutlinePosition[     32689,       1218]
CellTagsIndexPosition[     32596,       1212]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Calculus" :> 
         Documentation`HelpLookup["paclet:tutorial/CalculusOverview"]}, #, 
        2]& , Evaluator -> Automatic], {
      1->"\<\"Calculus\"\>"}, "\<\"related tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Integrals That Can and Cannot Be Done", "Title",
 CellTags->{"S3.5.7", "7.1"},
 CellID->28983],

Cell["\<\
Evaluating integrals is much more difficult than evaluating derivatives. For \
derivatives, there is a systematic procedure based on the chain rule that \
effectively allows any derivative to be worked out. But for integrals, there \
is no such systematic procedure. \
\>", "Text",
 CellID->5987],

Cell["\<\
One of the main problems is that it is difficult to know what kinds of \
functions will be needed to evaluate a particular integral. When you work out \
a derivative, you always end up with functions that are of the same kind or \
simpler than the ones you started with. But when you work out integrals, you \
often end up needing to use functions that are much more complicated than the \
ones you started with. \
\>", "Text",
 CellID->10823],

Cell["\<\
This integral can be evaluated using the same kind of functions that appeared \
in the input. \
\>", "MathCaption",
 CellID->397],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Log", "[", "x", "]"}], "^", "2"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[1]:= ",
 CellID->11096],

Cell[BoxData[
 RowBox[{
  RowBox[{"2", " ", "x"}], "-", 
  RowBox[{"2", " ", "x", " ", 
   RowBox[{"Log", "[", "x", "]"}]}], "+", 
  RowBox[{"x", " ", 
   SuperscriptBox[
    RowBox[{"Log", "[", "x", "]"}], "2"]}]}]], "Output",
 ImageSize->{166, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]= ",
 CellID->16594]
}, Open  ]],

Cell[TextData[{
 "But for this integral the special function ",
 Cell[BoxData[
  ButtonBox["LogIntegral",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LogIntegral"]], "InlineFormula"],
 " is needed. "
}], "MathCaption",
 CellID->8901],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{"Log", "[", "x", "]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:= ",
 CellID->17653],

Cell[BoxData[
 RowBox[{
  RowBox[{"x", " ", 
   RowBox[{"Log", "[", 
    RowBox[{"Log", "[", "x", "]"}], "]"}]}], "-", 
  RowBox[{"LogIntegral", "[", "x", "]"}]}]], "Output",
 ImageSize->{202, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]= ",
 CellID->14520]
}, Open  ]],

Cell["\<\
It is not difficult to find integrals that require all sorts of functions. \
\>", "MathCaption",
 CellID->30503],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"x", "^", "2"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[3]:= ",
 CellID->13472],

Cell[BoxData[
 RowBox[{
  SqrtBox[
   FractionBox["\[Pi]", "2"]], " ", 
  RowBox[{"FresnelS", "[", 
   RowBox[{
    SqrtBox[
     FractionBox["2", "\[Pi]"]], " ", "x"}], "]"}]}]], "Output",
 ImageSize->{151, 40},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]= ",
 CellID->20]
}, Open  ]],

Cell[TextData[{
 "This integral involves an incomplete gamma function. Note that the power is \
carefully set up to allow any complex value of ",
 Cell[BoxData["x"], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->5033],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{"x", "^", "a"}]}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[5]:= ",
 CellID->30057],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{"x", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      SuperscriptBox["x", "a"], ")"}], 
     RowBox[{
      RowBox[{"-", "1"}], "/", "a"}]], " ", 
    RowBox[{"Gamma", "[", 
     RowBox[{
      FractionBox["1", "a"], ",", 
      SuperscriptBox["x", "a"]}], "]"}]}], "a"]}]], "Output",
 ImageSize->{159, 38},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]= ",
 CellID->15665]
}, Open  ]],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " includes a very wide range of mathematical functions, and by using these \
functions a great many integrals can be done. But it is still possible to \
find even fairly simple\[Hyphen]looking integrals that just cannot be done in \
terms of any standard mathematical functions. "
}], "Text",
 CellID->2732],

Cell["\<\
Here is a fairly simple\[Hyphen]looking integral that cannot be done in terms \
of any standard mathematical functions. \
\>", "MathCaption",
 CellID->12280],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], "/", 
    RowBox[{"Log", "[", "x", "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[6]:= ",
 CellID->25167],

Cell[BoxData[
 RowBox[{"\[Integral]", 
  RowBox[{
   FractionBox[
    RowBox[{"Sin", "[", "x", "]"}], 
    RowBox[{"Log", "[", "x", "]"}]], 
   RowBox[{"\[DifferentialD]", "x"}]}]}]], "Output",
 ImageSize->{81, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]= ",
 CellID->16966]
}, Open  ]],

Cell["\<\
The main point of being able to do an integral in terms of standard \
mathematical functions is that it lets one use the known properties of these \
functions to evaluate or manipulate the result one gets. \
\>", "Text",
 CellID->25690],

Cell[TextData[{
 "In the most convenient cases, integrals can be done purely in terms of \
elementary functions such as exponentials, logarithms and trigonometric \
functions. In fact, if you give an integrand that involves only such \
elementary functions, then one of the important capabilities of ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 " is that if the corresponding integral can be expressed in terms of \
elementary functions, then ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 " will essentially always succeed in finding it."
}], "Text",
 CellID->868],

Cell["\<\
Integrals of rational functions are straightforward to evaluate, and always \
come out in terms of rational functions, logarithms and inverse trigonometric \
functions. \
\>", "MathCaption",
 CellID->29926],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"x", "/", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", "1"}], ")"}], 
      RowBox[{"(", 
       RowBox[{"x", "+", "2"}], ")"}]}], ")"}]}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[7]:= ",
 CellID->22828],

Cell[BoxData[
 RowBox[{
  RowBox[{
   FractionBox["1", "3"], " ", 
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"-", "1"}], "+", "x"}], "]"}]}], "+", 
  RowBox[{
   FractionBox["2", "3"], " ", 
   RowBox[{"Log", "[", 
    RowBox[{"2", "+", "x"}], "]"}]}]}]], "Output",
 ImageSize->{176, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]= ",
 CellID->21479]
}, Open  ]],

Cell["\<\
The integral here is still of the same form, but now involves an implicit sum \
over the roots of a polynomial. \
\>", "MathCaption",
 CellID->14144],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{"2", " ", "x"}], "+", 
      RowBox[{"x", "^", "3"}]}], ")"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[8]:= ",
 CellID->29703],

Cell[BoxData[
 RowBox[{"RootSum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"1", "+", 
     RowBox[{"2", " ", "#1"}], "+", 
     SuperscriptBox["#1", "3"]}], "&"}], ",", 
   RowBox[{
    FractionBox[
     RowBox[{"Log", "[", 
      RowBox[{"x", "-", "#1"}], "]"}], 
     RowBox[{"2", "+", 
      RowBox[{"3", " ", 
       SuperscriptBox["#1", "2"]}]}]], "&"}]}], "]"}]], "Output",
 ImageSize->{248, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]= ",
 CellID->9449]
}, Open  ]],

Cell["\<\
This finds numerical approximations to all the root objects. \
\>", "MathCaption",
 CellID->18833],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[9]:= ",
 CellID->12003],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "0.19107976295300613`"}], "-", 
     RowBox[{"0.08854101973827559`", " ", "\[ImaginaryI]"}]}], ")"}], " ", 
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "0.22669882575820188`"}], "-", 
       RowBox[{"1.4677115087102242`", " ", "\[ImaginaryI]"}]}], ")"}], "+", 
     "x"}], "]"}]}], "-", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{"0.19107976295300613`", "\[InvisibleSpace]", "-", 
     RowBox[{"0.08854101973827559`", " ", "\[ImaginaryI]"}]}], ")"}], " ", 
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", "0.22669882575820188`"}], "+", 
       RowBox[{"1.4677115087102242`", " ", "\[ImaginaryI]"}]}], ")"}], "+", 
     "x"}], "]"}]}], "+", 
  RowBox[{"0.38215952590601215`", " ", 
   RowBox[{"Log", "[", 
    RowBox[{"0.45339765151640377`", "\[InvisibleSpace]", "+", "x"}], 
    "]"}]}]}]], "Output",
 ImageSize->{537, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]= ",
 CellID->22910]
}, Open  ]],

Cell["\<\
Integrals of trigonometric functions usually come out in terms of other \
trigonometric functions. \
\>", "MathCaption",
 CellID->3240],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Sin", "[", "x", "]"}], "^", "3"}], " ", 
    RowBox[{
     RowBox[{"Cos", "[", "x", "]"}], "^", "2"}]}], ",", "x"}], "]"}]], "Input",\

 CellLabel->"In[10]:= ",
 CellID->21027],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox[
    RowBox[{"Cos", "[", "x", "]"}], "8"]}], "-", 
  RowBox[{
   FractionBox["1", "48"], " ", 
   RowBox[{"Cos", "[", 
    RowBox[{"3", " ", "x"}], "]"}]}], "+", 
  RowBox[{
   FractionBox["1", "80"], " ", 
   RowBox[{"Cos", "[", 
    RowBox[{"5", " ", "x"}], "]"}]}]}]], "Output",
 ImageSize->{230, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]= ",
 CellID->15610]
}, Open  ]],

Cell["\<\
This is a fairly simple integral involving algebraic functions. \
\>", "MathCaption",
 CellID->527],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "x", "]"}], " ", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"1", "+", "x"}], "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[11]:= ",
 CellID->15282],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "4"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     SqrtBox["x"], " ", 
     SqrtBox[
      RowBox[{"1", "+", "x"}]], " ", 
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{"2", " ", "x"}]}], ")"}]}], "-", 
    RowBox[{"ArcSinh", "[", 
     SqrtBox["x"], "]"}]}], ")"}]}]], "Output",
 ImageSize->{256, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]= ",
 CellID->26393]
}, Open  ]],

Cell["Here is an integral involving nested square roots. ", "MathCaption",
 CellID->17187],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", 
    RowBox[{"x", "+", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[12]:= ",
 CellID->7863],

Cell[BoxData[
 RowBox[{
  RowBox[{
   FractionBox["1", "12"], " ", 
   SqrtBox[
    RowBox[{
     SqrtBox["x"], "+", "x"}]], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "3"}], "+", 
     RowBox[{"2", " ", 
      SqrtBox["x"]}], "+", 
     RowBox[{"8", " ", "x"}]}], ")"}]}], "+", 
  RowBox[{
   FractionBox["1", "8"], " ", 
   RowBox[{"Log", "[", 
    RowBox[{"1", "+", 
     RowBox[{"2", " ", 
      SqrtBox["x"]}], "+", 
     RowBox[{"2", " ", 
      SqrtBox[
       RowBox[{
        SqrtBox["x"], "+", "x"}]]}]}], "]"}]}]}]], "Output",
 ImageSize->{396, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]= ",
 CellID->23961]
}, Open  ]],

Cell["\<\
By nesting elementary functions you sometimes get integrals that can be done \
in terms of elementary functions. \
\>", "MathCaption",
 CellID->2565],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Cos", "[", 
    RowBox[{"Log", "[", "x", "]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[13]:= ",
 CellID->28599],

Cell[BoxData[
 RowBox[{
  RowBox[{
   FractionBox["1", "2"], " ", "x", " ", 
   RowBox[{"Cos", "[", 
    RowBox[{"Log", "[", "x", "]"}], "]"}]}], "+", 
  RowBox[{
   FractionBox["1", "2"], " ", "x", " ", 
   RowBox[{"Sin", "[", 
    RowBox[{"Log", "[", "x", "]"}], "]"}]}]}]], "Output",
 ImageSize->{219, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]= ",
 CellID->19251]
}, Open  ]],

Cell["But more often other kinds of functions are needed. ", "MathCaption",
 CellID->27318],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{"Cos", "[", "x", "]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[14]:= ",
 CellID->12081],

Cell[BoxData[
 RowBox[{
  FractionBox[
   RowBox[{"\[ImaginaryI]", " ", 
    SuperscriptBox["x", "2"]}], "2"], "-", 
  RowBox[{"x", " ", 
   RowBox[{"Log", "[", 
    RowBox[{"1", "+", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"2", " ", "\[ImaginaryI]", " ", "x"}]]}], "]"}]}], "+", 
  RowBox[{"x", " ", 
   RowBox[{"Log", "[", 
    RowBox[{"Cos", "[", "x", "]"}], "]"}]}], "+", 
  RowBox[{
   FractionBox["1", "2"], " ", "\[ImaginaryI]", " ", 
   RowBox[{"PolyLog", "[", 
    RowBox[{"2", ",", 
     RowBox[{"-", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"2", " ", "\[ImaginaryI]", " ", "x"}]]}]}], 
    "]"}]}]}]], "Output",
 ImageSize->{389, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]= ",
 CellID->3558]
}, Open  ]],

Cell["\<\
Integrals like this typically come out in terms of elliptic functions. \
\>", "MathCaption",
 CellID->28103],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", 
    RowBox[{"Cos", "[", "x", "]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[15]:= ",
 CellID->6358],

Cell[BoxData[
 RowBox[{"2", " ", 
  RowBox[{"EllipticE", "[", 
   RowBox[{
    FractionBox["x", "2"], ",", "2"}], "]"}]}]], "Output",
 ImageSize->{121, 28},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]= ",
 CellID->15238]
}, Open  ]],

Cell["\<\
But occasionally one can get results in terms of elementary functions alone. \
\
\>", "MathCaption",
 CellID->7178],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", 
    RowBox[{"Tan", "[", "x", "]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[16]:= ",
 CellID->17464],

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   RowBox[{"2", " ", 
    SqrtBox["2"]}]], 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", 
     RowBox[{"ArcTan", "[", 
      RowBox[{"1", "-", 
       RowBox[{
        SqrtBox["2"], " ", 
        SqrtBox[
         RowBox[{"Tan", "[", "x", "]"}]]}]}], "]"}]}], "+", 
    RowBox[{"2", " ", 
     RowBox[{"ArcTan", "[", 
      RowBox[{"1", "+", 
       RowBox[{
        SqrtBox["2"], " ", 
        SqrtBox[
         RowBox[{"Tan", "[", "x", "]"}]]}]}], "]"}]}], "+", 
    RowBox[{"Log", "[", 
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      RowBox[{
       SqrtBox["2"], " ", 
       SqrtBox[
        RowBox[{"Tan", "[", "x", "]"}]]}], "-", 
      RowBox[{"Tan", "[", "x", "]"}]}], "]"}], "-", 
    RowBox[{"Log", "[", 
     RowBox[{"1", "+", 
      RowBox[{
       SqrtBox["2"], " ", 
       SqrtBox[
        RowBox[{"Tan", "[", "x", "]"}]]}], "+", 
      RowBox[{"Tan", "[", "x", "]"}]}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{443, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]= ",
 CellID->17457]
}, Open  ]],

Cell[TextData[{
 "Integrals like this can systematically be done using ",
 Cell[BoxData[
  ButtonBox["Piecewise",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Piecewise"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->6572],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"2", "^", 
    RowBox[{"Max", "[", 
     RowBox[{"x", ",", 
      RowBox[{"1", "-", "x"}]}], "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[17]:= ",
 CellID->7221],

Cell[BoxData[
 RowBox[{"\[Piecewise]", GridBox[{
    {
     RowBox[{"-", 
      FractionBox[
       SuperscriptBox["2", 
        RowBox[{"1", "-", "x"}]], 
       RowBox[{"Log", "[", "2", "]"}]]}], 
     RowBox[{"x", "\[LessEqual]", 
      FractionBox["1", "2"]}]},
    {
     RowBox[{
      RowBox[{"-", 
       FractionBox[
        RowBox[{"2", " ", 
         SqrtBox["2"]}], 
        RowBox[{"Log", "[", "2", "]"}]]}], "+", 
      FractionBox[
       SuperscriptBox["2", "x"], 
       RowBox[{"Log", "[", "2", "]"}]]}], 
     TagBox["True",
      "PiecewiseDefault",
      AutoDelete->False,
      DeletionWarning->True]}
   },
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}},
   GridBoxSpacings->{
    "Columns" -> {0.28, {0.84}, 0.28}, "ColumnsIndexed" -> {}, 
     "Rows" -> {0.2, {0.4}, 0.2}, "RowsIndexed" -> {}}]}]], "Output",
 ImageSize->{147, 63},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]= ",
 CellID->16922]
}, Open  ]],

Cell[TextData[{
 "Beyond working with elementary functions, ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 " includes a large number of algorithms for dealing with special functions. \
Sometimes it uses a direct generalization of the procedure for elementary \
functions. But more often its strategy is first to try to write the integrand \
in a form that can be integrated in terms of certain sophisticated special \
functions, and then having done this to try to find reductions of these \
sophisticated functions to more familiar functions. "
}], "Text",
 CellID->25756],

Cell["\<\
To integrate this Bessel function requires a generalized hypergeometric \
function. \
\>", "MathCaption",
 CellID->10867],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"BesselJ", "[", 
    RowBox[{"0", ",", "x"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[18]:= ",
 CellID->21260],

Cell[BoxData[
 RowBox[{"x", " ", 
  RowBox[{"HypergeometricPFQ", "[", 
   RowBox[{
    RowBox[{"{", 
     FractionBox["1", "2"], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      FractionBox["3", "2"]}], "}"}], ",", 
    RowBox[{"-", 
     FractionBox[
      SuperscriptBox["x", "2"], "4"]}]}], "]"}]}]], "Output",
 ImageSize->{263, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]= ",
 CellID->9298]
}, Open  ]],

Cell["\<\
Sometimes the integrals can be reduced to more familiar forms. \
\>", "MathCaption",
 CellID->24019],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "3"}], " ", 
    RowBox[{"BesselJ", "[", 
     RowBox[{"0", ",", "x"}], "]"}]}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[20]:= ",
 CellID->25352],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   SuperscriptBox["x", "2"]}], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{
     RowBox[{"-", "2"}], " ", 
     RowBox[{"BesselJ", "[", 
      RowBox[{"2", ",", "x"}], "]"}]}], "+", 
    RowBox[{"x", " ", 
     RowBox[{"BesselJ", "[", 
      RowBox[{"3", ",", "x"}], "]"}]}]}], ")"}]}]], "Output",
 ImageSize->{258, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]= ",
 CellID->14062]
}, Open  ]],

Cell[TextData[{
 "A large book of integral tables will list perhaps a few thousand indefinite \
integrals. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can do essentially all of these integrals. And because it contains general \
algorithms rather than just specific cases, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can actually do a vastly wider range of integrals."
}], "Text",
 CellID->8411],

Cell["\<\
You could expect to find this integral in any large book of integral tables. \
\
\>", "MathCaption",
 CellID->16811],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Log", "[", 
     RowBox[{"1", "-", "x"}], "]"}], "/", "x"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[21]:= ",
 CellID->21793],

Cell[BoxData[
 RowBox[{"-", 
  RowBox[{"PolyLog", "[", 
   RowBox[{"2", ",", "x"}], "]"}]}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]= ",
 CellID->17405]
}, Open  ]],

Cell["\<\
To do this integral, however, requires a more general algorithm, rather than \
just a direct table lookup. \
\>", "MathCaption",
 CellID->6881],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Log", "[", 
     RowBox[{"1", "+", 
      RowBox[{"3", " ", "x"}], "+", 
      RowBox[{"x", "^", "2"}]}], "]"}], "/", "x"}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[22]:= ",
 CellID->3651],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Log", "[", "x", "]"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Log", "[", 
      RowBox[{
       RowBox[{
        FractionBox["1", "2"], " ", 
        RowBox[{"(", 
         RowBox[{"3", "-", 
          SqrtBox["5"]}], ")"}]}], "+", "x"}], "]"}], "-", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "+", 
       FractionBox[
        RowBox[{"2", " ", "x"}], 
        RowBox[{"3", "-", 
         SqrtBox["5"]}]]}], "]"}]}], ")"}]}], "+", 
  RowBox[{
   RowBox[{"Log", "[", "x", "]"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"Log", "[", 
      RowBox[{
       RowBox[{
        FractionBox["1", "2"], " ", 
        RowBox[{"(", 
         RowBox[{"3", "+", 
          SqrtBox["5"]}], ")"}]}], "+", "x"}], "]"}], "-", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "+", 
       FractionBox[
        RowBox[{"2", " ", "x"}], 
        RowBox[{"3", "+", 
         SqrtBox["5"]}]]}], "]"}]}], ")"}]}], "+", 
  RowBox[{
   RowBox[{"Log", "[", "x", "]"}], " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      RowBox[{"Log", "[", 
       RowBox[{
        RowBox[{
         FractionBox["1", "2"], " ", 
         RowBox[{"(", 
          RowBox[{"3", "-", 
           SqrtBox["5"]}], ")"}]}], "+", "x"}], "]"}]}], "-", 
     RowBox[{"Log", "[", 
      RowBox[{
       RowBox[{
        FractionBox["1", "2"], " ", 
        RowBox[{"(", 
         RowBox[{"3", "+", 
          SqrtBox["5"]}], ")"}]}], "+", "x"}], "]"}], "+", 
     RowBox[{"Log", "[", 
      RowBox[{"1", "+", 
       RowBox[{"3", " ", "x"}], "+", 
       SuperscriptBox["x", "2"]}], "]"}]}], ")"}]}], "-", 
  RowBox[{"PolyLog", "[", 
   RowBox[{"2", ",", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"2", " ", "x"}], 
      RowBox[{"3", "-", 
       SqrtBox["5"]}]]}]}], "]"}], "-", 
  RowBox[{"PolyLog", "[", 
   RowBox[{"2", ",", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"2", " ", "x"}], 
      RowBox[{"3", "+", 
       SqrtBox["5"]}]]}]}], "]"}]}]], "Output",
 ImageSize->{462, 139},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]= ",
 CellID->25760]
}, Open  ]],

Cell[TextData[{
 "Particularly if you introduce new mathematical functions of your own, you \
may want to teach ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " new kinds of integrals. You can do this by making appropriate definitions \
for ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->17898],

Cell[TextData[{
 "In the case of differentiation, the chain rule allows one to reduce all \
derivatives to a standard form, represented in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " using ",
 Cell[BoxData[
  ButtonBox["Derivative",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Derivative"]], "InlineFormula"],
 ". But for integration, no such similar standard form exists, and as a \
result you often have to make definitions for several different versions of \
the same integral. Changes of variables and other transformations can rarely \
be done automatically by ",
 Cell[BoxData[
  ButtonBox["Integrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Integrate"]], "InlineFormula"],
 "."
}], "Text",
 CellID->21271],

Cell[TextData[{
 "This integral cannot be done in terms of any of the standard mathematical \
functions built into ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ". "
}], "MathCaption",
 CellID->15842],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"Sin", "[", "x", "]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[23]:= ",
 CellID->5694],

Cell[BoxData[
 RowBox[{"\[Integral]", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"Sin", "[", "x", "]"}], "]"}], 
   RowBox[{"\[DifferentialD]", "x"}]}]}]], "Output",
 ImageSize->{112, 29},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]= ",
 CellID->5084]
}, Open  ]],

Cell["\<\
Before you add your own rules for integration, you have to remove write \
protection. \
\>", "MathCaption",
 CellID->21535],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Unprotect", "[", "Integrate", "]"}]], "Input",
 CellLabel->"In[24]:= ",
 CellID->16500],

Cell[BoxData[
 RowBox[{"{", "\<\"Integrate\"\>", "}"}]], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]= ",
 CellID->10955]
}, Open  ]],

Cell["\<\
You can set up your own rule to define the integral to be, say, a \
\[OpenCurlyDoubleQuote]Jones\[CloseCurlyDoubleQuote] function. \
\>", "MathCaption",
 CellID->31155],

Cell[BoxData[
 RowBox[{
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"Sin", "[", 
      RowBox[{"a_.", "+", 
       RowBox[{"b_.", " ", "x_"}]}], "]"}], "]"}], ",", "x_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"Jones", "[", 
    RowBox[{"a", ",", "x"}], "]"}], "/", "b"}]}]], "Input",
 CellLabel->"In[25]:= ",
 CellID->20323],

Cell[TextData[{
 "Now ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can do integrals that give Jones functions. "
}], "MathCaption",
 CellID->650],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{"Sin", "[", 
     RowBox[{"3", "x"}], "]"}], "]"}], ",", "x"}], "]"}]], "Input",
 CellLabel->"In[26]:= ",
 CellID->6688],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "3"], " ", 
  RowBox[{"Jones", "[", 
   RowBox[{"0", ",", "x"}], "]"}]}]], "Output",
 ImageSize->{93, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]= ",
 CellID->5939]
}, Open  ]],

Cell[TextData[{
 "As it turns out, the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Integral]", 
    RowBox[{
     RowBox[{"sin", "(", 
      RowBox[{"sin", "(", "x", ")"}], ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " can in principle be represented as an infinite sum of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Null]", "2"], 
    SubscriptBox["F", "1"]}], TraditionalForm]], "InlineMath"],
 " hypergeometric functions, or as a suitably generalized Kamp\[EAcute] de F\
\[EAcute]riet hypergeometric function of two variables."
}], "Text",
 CellID->4010],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->877461274],

Cell[TextData[ButtonBox["Calculus",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/CalculusOverview"]], "RelatedTutorials",
 CellID->808424779]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Integrals That Can and Cannot Be Done - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox["Calculus", "OverviewNavText"], BaseStyle -> "Link", 
        ButtonData -> "paclet:tutorial/CalculusOverview"]], "Text", 
      FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/IndefiniteIntegrals"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/DefiniteIntegrals"]}], "Text",
       FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 37, 38.5211212}", "context" -> "", 
    "keywords" -> {
     "Dwights tables", "GradshteynRyzhik", "Hypergeometric functions", 
      "integrals", "integration", "Jones functions", 
      "Kampe de Feriet functions", "mathematical tables", 
      "rational expressions", "Risch algorithm", "RootSum", "sine integrals"},
     "index" -> True, "label" -> "Mathematica Tutorial", "language" -> "en", 
    "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "Evaluating integrals is much more difficult than evaluating derivatives. \
For derivatives, there is a systematic procedure based on the chain rule that \
effectively allows any derivative to be worked out. But for integrals, there \
is no such systematic procedure. One of the main problems is that it is \
difficult to know what kinds of functions will be needed to evaluate a \
particular integral. When you work out a derivative, you always end up with \
functions that are of the same kind or simpler than the ones you started \
with. But when you work out integrals, you often end up needing to use \
functions that are much more complicated than the ones you started with. This \
integral can be evaluated using the same kind of functions that appeared in \
the input. ", "synonyms" -> {"3.5.7"}, "title" -> 
    "Integrals That Can and Cannot Be Done", "type" -> "Tutorial", "uri" -> 
    "tutorial/IntegralsThatCanAndCannotBeDone"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S3.5.7"->{
  Cell[1215, 42, 100, 2, 70, "Title",
   CellTags->{"S3.5.7", "7.1"},
   CellID->28983]},
 "7.1"->{
  Cell[1215, 42, 100, 2, 70, "Title",
   CellTags->{"S3.5.7", "7.1"},
   CellID->28983]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S3.5.7", 32383, 1201},
 {"7.1", 32483, 1205}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 590, 15, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1215, 42, 100, 2, 70, "Title",
 CellTags->{"S3.5.7", "7.1"},
 CellID->28983],
Cell[1318, 46, 306, 6, 70, "Text",
 CellID->5987],
Cell[1627, 54, 453, 8, 70, "Text",
 CellID->10823],
Cell[2083, 64, 139, 4, 70, "MathCaption",
 CellID->397],
Cell[CellGroupData[{
Cell[2247, 72, 181, 6, 28, "Input",
 CellID->11096],
Cell[2431, 80, 355, 12, 36, "Output",
 CellID->16594]
}, Open  ]],
Cell[2801, 95, 238, 8, 70, "MathCaption",
 CellID->8901],
Cell[CellGroupData[{
Cell[3064, 107, 188, 6, 28, "Input",
 CellID->17653],
Cell[3255, 115, 302, 10, 36, "Output",
 CellID->14520]
}, Open  ]],
Cell[3572, 128, 122, 3, 70, "MathCaption",
 CellID->30503],
Cell[CellGroupData[{
Cell[3719, 135, 181, 6, 28, "Input",
 CellID->13472],
Cell[3903, 143, 314, 12, 61, "Output",
 CellID->20]
}, Open  ]],
Cell[4232, 158, 223, 6, 70, "MathCaption",
 CellID->5033],
Cell[CellGroupData[{
Cell[4480, 168, 202, 7, 70, "Input",
 CellID->30057],
Cell[4685, 177, 469, 17, 59, "Output",
 CellID->15665]
}, Open  ]],
Cell[5169, 197, 373, 8, 70, "Text",
 CellID->2732],
Cell[5545, 207, 167, 4, 70, "MathCaption",
 CellID->12280],
Cell[CellGroupData[{
Cell[5737, 215, 213, 7, 70, "Input",
 CellID->25167],
Cell[5953, 224, 320, 11, 53, "Output",
 CellID->16966]
}, Open  ]],
Cell[6288, 238, 246, 5, 70, "Text",
 CellID->25690],
Cell[6537, 245, 723, 17, 70, "Text",
 CellID->868],
Cell[7263, 264, 216, 5, 70, "MathCaption",
 CellID->29926],
Cell[CellGroupData[{
Cell[7504, 273, 307, 12, 70, "Input",
 CellID->22828],
Cell[7814, 287, 401, 15, 51, "Output",
 CellID->21479]
}, Open  ]],
Cell[8230, 305, 159, 4, 70, "MathCaption",
 CellID->14144],
Cell[CellGroupData[{
Cell[8414, 313, 264, 9, 70, "Input",
 CellID->29703],
Cell[8681, 324, 506, 18, 53, "Output",
 CellID->9449]
}, Open  ]],
Cell[9202, 345, 108, 3, 70, "MathCaption",
 CellID->18833],
Cell[CellGroupData[{
Cell[9335, 352, 94, 3, 70, "Input",
 CellID->12003],
Cell[9432, 357, 1099, 33, 54, "Output",
 CellID->22910]
}, Open  ]],
Cell[10546, 393, 145, 4, 70, "MathCaption",
 CellID->3240],
Cell[CellGroupData[{
Cell[10716, 401, 268, 10, 70, "Input",
 CellID->21027],
Cell[10987, 413, 465, 17, 51, "Output",
 CellID->15610]
}, Open  ]],
Cell[11467, 433, 109, 3, 70, "MathCaption",
 CellID->527],
Cell[CellGroupData[{
Cell[11601, 440, 242, 8, 70, "Input",
 CellID->15282],
Cell[11846, 450, 471, 18, 51, "Output",
 CellID->26393]
}, Open  ]],
Cell[12332, 471, 90, 1, 70, "MathCaption",
 CellID->17187],
Cell[CellGroupData[{
Cell[12447, 476, 216, 7, 70, "Input",
 CellID->7863],
Cell[12666, 485, 677, 27, 53, "Output",
 CellID->23961]
}, Open  ]],
Cell[13358, 515, 159, 4, 70, "MathCaption",
 CellID->2565],
Cell[CellGroupData[{
Cell[13542, 523, 189, 6, 70, "Input",
 CellID->28599],
Cell[13734, 531, 415, 14, 51, "Output",
 CellID->19251]
}, Open  ]],
Cell[14164, 548, 91, 1, 70, "MathCaption",
 CellID->27318],
Cell[CellGroupData[{
Cell[14280, 553, 189, 6, 70, "Input",
 CellID->12081],
Cell[14472, 561, 781, 25, 54, "Output",
 CellID->3558]
}, Open  ]],
Cell[15268, 589, 118, 3, 70, "MathCaption",
 CellID->28103],
Cell[CellGroupData[{
Cell[15411, 596, 189, 6, 70, "Input",
 CellID->6358],
Cell[15603, 604, 262, 9, 49, "Output",
 CellID->15238]
}, Open  ]],
Cell[15880, 616, 125, 4, 70, "MathCaption",
 CellID->7178],
Cell[CellGroupData[{
Cell[16030, 624, 190, 6, 70, "Input",
 CellID->17464],
Cell[16223, 632, 1119, 41, 77, "Output",
 CellID->17457]
}, Open  ]],
Cell[17357, 676, 234, 8, 70, "MathCaption",
 CellID->6572],
Cell[CellGroupData[{
Cell[17616, 688, 234, 8, 70, "Input",
 CellID->7221],
Cell[17853, 698, 1041, 35, 84, "Output",
 CellID->16922]
}, Open  ]],
Cell[18909, 736, 651, 13, 70, "Text",
 CellID->25756],
Cell[19563, 751, 131, 4, 70, "MathCaption",
 CellID->10867],
Cell[CellGroupData[{
Cell[19719, 759, 186, 6, 70, "Input",
 CellID->21260],
Cell[19908, 767, 453, 16, 54, "Output",
 CellID->9298]
}, Open  ]],
Cell[20376, 786, 110, 3, 70, "MathCaption",
 CellID->24019],
Cell[CellGroupData[{
Cell[20511, 793, 237, 8, 70, "Input",
 CellID->25352],
Cell[20751, 803, 468, 17, 36, "Output",
 CellID->14062]
}, Open  ]],
Cell[21234, 823, 416, 11, 70, "Text",
 CellID->8411],
Cell[21653, 836, 126, 4, 70, "MathCaption",
 CellID->16811],
Cell[CellGroupData[{
Cell[21804, 844, 208, 7, 70, "Input",
 CellID->21793],
Cell[22015, 853, 232, 8, 36, "Output",
 CellID->17405]
}, Open  ]],
Cell[22262, 864, 153, 4, 70, "MathCaption",
 CellID->6881],
Cell[CellGroupData[{
Cell[22440, 872, 274, 10, 70, "Input",
 CellID->3651],
Cell[22717, 884, 2132, 77, 160, "Output",
 CellID->25760]
}, Open  ]],
Cell[24864, 964, 404, 13, 70, "Text",
 CellID->17898],
Cell[25271, 979, 738, 20, 70, "Text",
 CellID->21271],
Cell[26012, 1001, 207, 7, 70, "MathCaption",
 CellID->15842],
Cell[CellGroupData[{
Cell[26244, 1012, 188, 6, 70, "Input",
 CellID->5694],
Cell[26435, 1020, 298, 10, 50, "Output",
 CellID->5084]
}, Open  ]],
Cell[26748, 1033, 133, 4, 70, "MathCaption",
 CellID->21535],
Cell[CellGroupData[{
Cell[26906, 1041, 111, 3, 70, "Input",
 CellID->16500],
Cell[27020, 1046, 194, 6, 36, "Output",
 CellID->10955]
}, Open  ]],
Cell[27229, 1055, 178, 4, 70, "MathCaption",
 CellID->31155],
Cell[27410, 1061, 358, 12, 70, "Input",
 CellID->20323],
Cell[27771, 1075, 155, 6, 70, "MathCaption",
 CellID->650],
Cell[CellGroupData[{
Cell[27951, 1085, 209, 7, 70, "Input",
 CellID->6688],
Cell[28163, 1094, 254, 9, 51, "Output",
 CellID->5939]
}, Open  ]],
Cell[28432, 1106, 625, 18, 70, "Text",
 CellID->4010],
Cell[CellGroupData[{
Cell[29082, 1128, 72, 1, 70, "RelatedTutorialsSection",
 CellID->877461274],
Cell[29157, 1131, 146, 3, 70, "RelatedTutorials",
 CellID->808424779]
}, Open  ]]
}, Open  ]],
Cell[29330, 1138, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

