(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    619781,      13481]
NotebookOptionsPosition[    596484,      12665]
NotebookOutlinePosition[    602729,      12870]
CellTagsIndexPosition[    601822,      12834]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Linear Algebra in Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/LinearAlgebraOverview"], 
          "Matrix and Tensor Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations"], 
          "Working with Sparse Arrays" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraSparseArrays"], 
          "Matrix Computations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraMatrixComputations"], "Matrix Types" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraMatrixTypes"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Linear Algebra in Mathematica\"\>", 
       2->"\<\"Matrix and Tensor Operations\"\>", 
       3->"\<\"Working with Sparse Arrays\"\>", 
       4->"\<\"Matrix Computations\"\>", 
       5->"\<\"Matrix Types\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "Linear Systems" :> 
          Documentation`HelpLookup["paclet:guide/LinearSystems"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Matrices and Linear Algebra\"\>", 
       2->"\<\"Linear Systems\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Linear Algebra Examples", "Title",
 ShowCellBracket->True,
 CellTags->{"t:58", "c:152"},
 CellID->349852864],

Cell[TextData[{
 "This tutorial shows a number of examples of the use of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " for computations that involve linear algebra."
}], "Text",
 CellID->166859762],

Cell[CellGroupData[{

Cell["Matrix Ordering", "Section",
 CellTags->{"t:151", "c:153"},
 CellID->1945860599],

Cell["\<\
Certain sparse matrix techniques try to reorder the matrix so that elements \
are grouped into blocks. The computation then works on each block using dense \
matrix techniques. One simple way to order a matrix into blocks involves \
sorting according to the sum of elements on each row. This will be \
demonstrated in this example.\
\>", "Text",
 CellID->50627230],

Cell["First, generate a symmetric random sparse matrix.", "Text",
 CellID->488330398],

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "30"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"nz", "=", "30"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sp", "=", 
   RowBox[{"SparseArray", "[", "\[IndentingNewLine]", "\t", 
    RowBox[{
     RowBox[{
      RowBox[{"RandomInteger", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "n"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"nz", ",", "2"}], "}"}]}], "]"}], "\[Rule]", 
      RowBox[{"Table", "[", 
       RowBox[{"1", ",", 
        RowBox[{"{", "nz", "}"}]}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sp", "=", 
   RowBox[{"sp", "+", 
    RowBox[{"Transpose", "[", "sp", "]"}]}]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1833983999],

Cell["Here is a plot of the matrix.", "Text",
 CellID->998430637],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", "sp", "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->186498401],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWsFu00AQ3cR2SlpRGgSCHkBREaStCggOSFzKAQpcypELpxBVtBIIKL1w
4oi4Iw58Rw/83bKznknXs7t2HDeJaztSdjL2ezP2emy/3ez+8OTw4NPw5Gg0
7L86Hn45PBp967/8fKw2BS0hxKn6/rsp1G+pfunvqZTK2m1PtYEC3EPbVrat
7JayIfNhf4B+T7OvQgubpf7oKCI882O+iRhAE8X+uLVQW4bv2r86ZfbNTMSs
MmfFHWC/tq1IazPKmN0X056rRnRMRJG+7mHPUA9RBW4z312hvrq/rNq/IrbY
Gpn5kWwz/wHz7zM/YP7KHLOYfRAxxB8ewcAGMTvjmnD/lmqinY2VZbQHKcdI
8eAR1QK7oIx4qmKcLVHNoYzriOcZyWR92Uy9xbe9wBm5Iy66F9Pq7D1Gfcij
WvXG7wkf030HlQf9BNHvKo0G5Btk7NaKQffJz0qjd6T7aVAt9DVEd9FuoH1d
cfYHRHOdUS30R0TfrTT6F6IvlRxtjiG5SvAxTQ5KtU5szDZL4Wit2x37dgCH
psmv3biiPi+tBu5vtJEDX0z9WeOAzAyTjOecWby96cljaHOfutxTTWho191k
HpkcF7p1pjNGZq2XnzVANNf39Kbgo/kmSp4o63hNNtHu15pFqov6cB1tv2Gn
sOFptqzQz5AVoX3asFPY/I1VFtQV42zgzvleaxY9R0ix8/31YpWlZvWUpqCP
T40ztSaT/wfFKiq/PobeeoG9Rm8XGsXy+etJtPNFOQKt21MyJ69JWKEzn+UR
TDI6gPhdFZ9mNqcZHThjZN595Wd9lXmUykVFk7b/Id2VVi9W4k3++Po4CljQ
Y6voP2+izTAajT/3ZJ5KLj+LlNFtZL2tNYtmF+4g+xH6Nxp2CptmuiabdVgc
GiahA1JrPpaloJc0d4lx+LOb5hToX0k+H8WVVWfOUcHl0dY8/THvYyseFVqX
ljyXFTqW9tT/d8xxzdHZOiffaqhmLaAL0awFLH6u5V4LKFr/AdNiO04=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 184},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->346437503]
}, Open  ]],

Cell[TextData[{
 "Now compute a permutation of the matrix and then apply it to the matrix. \
This topic is discussed in \"",
 ButtonBox["Matrix Permutations",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#1318022893"],
 "\"."
}], "Text",
 CellID->343119060],

Cell[BoxData[{
 RowBox[{
  RowBox[{"perm", "=", 
   RowBox[{"Reverse", "[", 
    RowBox[{"Ordering", "[", 
     RowBox[{"sp", ".", 
      RowBox[{"Table", "[", 
       RowBox[{"1", ",", 
        RowBox[{"{", "n", "}"}]}], "]"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"spOrdered", "=", 
   RowBox[{"sp", "[", 
    RowBox[{"[", 
     RowBox[{"perm", ",", "perm"}], "]"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[7]:=",
 CellID->808449225],

Cell["\<\
Here is a plot of the reordered matrix; the effect of the reordering can be \
seen.\
\>", "Text",
 CellID->477090006],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", "spOrdered", "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->1895276591],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWktv00AQ3sR2SlpBCQJRJA5RJQitWhAIVUJCPATicagQF8SRqKrUVqDS
EnHjwoU/gDjwO3rg3y0765l0d7wb282jju1I2clsvm8mnoydz5vd7g/2dr/0
B/s7/e7r4/7Xvf2db91Xh8dqKmgIIU7U89+KUK+leqWfJ1Iqmxw7agwU4Dba
prJNZdeUDZkP7wfodzT7CowwLfVDRxHhqR/zTUQPhij2h2MCtWb4rvcvnTH7
nVTEtDKnxe1hXZuJSJenlDG9Fmc9Vo1omYhxat3BylCFqAPXme/uUF/fX1Tj
XxFbHI3M/JOsM/8u8zeZHzB/aYZZzBpEDPGHRzCwQcxO+U64n4jo4MMlqQF2
RhnwUMQwutWtoYz75Le0+yWJ1DO++Ryf2B1h1lUZry9473KGu8NrVD7Uohqi
jdWlX6VGX0X0j1KjPyL6UanRnxF9a87QR4julhq9jOifpUavIvqw4Gjzfor/
AvuYJgdlTSs25pimFp5h9PfgtIfzyUAO3ZBf/1jZpqCDwKUc79CPHLzxFJbv
KDLozwz3PiOzeavtyWfoXZ/C+66GUMW/aceX9j2UW+tZ3NRzoDhouoqsoN2t
NAuQW8j4UCkG6bYQ7aDSLPilWVTog1KjoR6kZB/OOeMtMl5UjhHJPPeVpWHI
uNOLjtZLaYIePmVrxbCURyC4ts2vNamy1FtPXPgUzVnkrKRtzWyPcT6c82Ob
dNYsOpi0030ZK8ltO67MooedMTKrtuKyaBVvC1k30L9Xs0exFfIpMt5UikHn
LF3hs62LlpUFXQPds4FsstcrwqazqIeWrkH8X806Sp4otC4IUaAXX6LP/5Wv
2bz2bYP9aU5ZsNQ7VHpp7IS2XtAxFhiX7xqIHlwTtJIEKwrLyral+06Nq7bW
OWbQynhE5Oee2p3nZ55sBhhdGnci+0oSmviCdPXx9HbKnO7O8e3hqXewuRD1
Drbxj7XYO9hE4z+2mpsO\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 184},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->638786328]
}, Open  ]],

Cell["\<\
Here is a much larger matrix with no structure and many random elements.\
\>", "Text",
 CellID->1502509129],

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "10000"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"nz", "=", "2000"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sp", "=", 
   RowBox[{"SparseArray", "[", "\[IndentingNewLine]", "\t", 
    RowBox[{
     RowBox[{
      RowBox[{"RandomInteger", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "n"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"nz", ",", "2"}], "}"}]}], "]"}], "\[Rule]", 
      RowBox[{"Table", "[", 
       RowBox[{"1", ",", 
        RowBox[{"{", "nz", "}"}]}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"n", ",", "n"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sp", "=", 
   RowBox[{"sp", "+", 
    RowBox[{"Transpose", "[", "sp", "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"perm", "=", 
   RowBox[{"Reverse", "[", 
    RowBox[{"Ordering", "[", 
     RowBox[{"sp", ".", 
      RowBox[{"Table", "[", 
       RowBox[{"1", ",", 
        RowBox[{"{", "n", "}"}]}], "]"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"spOrdered", "=", 
   RowBox[{"sp", "[", 
    RowBox[{"[", 
     RowBox[{"perm", ",", "perm"}], "]"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[10]:=",
 CellID->410726030],

Cell["This is a plot of the original matrix.", "Text",
 CellID->1059944752],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", " ", "sp", "]"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->1114594999],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->407509172]
}, Open  ]],

Cell["\<\
This plot of the reordered matrix shows that there are many rows and columns \
with no elements.\
\>", "Text",
 CellID->1812249831],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", "spOrdered", "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->722344216],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->406068805]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Full Rank Least Squares Solutions", "Section",
 CellTags->{"t:191", "c:154"},
 CellID->967762464],

Cell[TextData[{
 "Solving a matrix equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"A", ".", "X"}], "\[LongEqual]", "B"}], TraditionalForm]]],
 " is one of the key problems of matrix computation; techniques for solving \
it are discussed under \"",
 ButtonBox["Solving Linear Systems",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixComputations#81663253"],
 "\". If ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]]],
 " is an ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "\[Cross]", "n"}], TraditionalForm]]],
 " matrix, this is a set of ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]]],
 " linear equations in ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]]],
 " unknowns. If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", " ", ">", " ", "n"}], TraditionalForm]]],
 ", there are more equations than unknowns and the system is overdetermined. \
A general technique for finding least squares solutions is given under \"",
 ButtonBox["Least Squares Solutions",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixComputations#563250646"],
 "\". This example will show some different techniques for finding \
approximate solutions to overdetermined systems for full rank matrices."
}], "Text",
 CellID->142590070],

Cell["\<\
Note that all the example solutions in this section will work just as well if \
they were given sparse matrices as input.\
\>", "Text",
 CellID->2067040085],

Cell[CellGroupData[{

Cell["Least Squares Cholesky", "Subsection",
 CellTags->{"t:105", "c:155"},
 CellID->899751076],

Cell[TextData[{
 "This technique uses a ",
 ButtonBox["Cholesky decomposition",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixComputations#1567526616"],
 " to find a least squares solution. If a matrix ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]]],
 " has full rank, the solution can be found in the following way."
}], "Text",
 CellID->1683975805],

Cell[TextData[{
 "Solve A . x \[LongEqual] B\n\tC = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["A", "T"], ".", "A"}], TraditionalForm]]],
 "\n\td = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["A", "T"], ".", "B"}], TraditionalForm]]],
 "\n\tfind G such that C = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["G", "T"], ".", "G"}], TraditionalForm]]],
 "  (cholesky decomposition)\n\tfind y such that ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["G", "T"], TraditionalForm]]],
 ".y \[LongEqual] d\n\tthen G.x \[LongEqual] y"
}], "NumberedEquation",
 CellID->894370822],

Cell[TextData[{
 "These steps can be placed into a ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " function as follows."
}], "Text",
 CellID->1452360001],

Cell[BoxData[
 RowBox[{
  RowBox[{"L2CholeskySolve", "[", 
   RowBox[{"a_", ",", "b_"}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", " ", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"aT", ",", "c", ",", "d", ",", "g", ",", "gT", ",", "y"}], "}"}],
     ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"aT", " ", "=", " ", 
      RowBox[{"Transpose", "[", "a", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"c", "=", " ", 
      RowBox[{"aT", ".", "a"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"d", "=", 
      RowBox[{"aT", ".", "b"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"g", "=", 
      RowBox[{"CholeskyDecomposition", "[", "c", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"gT", "=", 
      RowBox[{"Transpose", "[", "g", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"y", "=", 
      RowBox[{"LinearSolve", "[", 
       RowBox[{"gT", ",", "d"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"LinearSolve", "[", 
      RowBox[{"g", ",", "y"}], "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1515405005],

Cell["The function can be applied as follows.", "Text",
 CellID->255865615],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.", ",", "2."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.", ",", "6."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4.5", ",", "6."}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"{", 
    RowBox[{"5.", ",", "6.", ",", "8."}], "}"}]}], ";"}], "\n", 
 RowBox[{"sol1", "=", 
  RowBox[{"L2CholeskySolve", "[", 
   RowBox[{"mat", ",", "vec"}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->1071130011],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4.411764705882248`"}], ",", "4.661764705882271`"}], 
  "}"}]], "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->511228008]
}, Open  ]],

Cell["The solution is close to satisfying the equation.", "Text",
 CellID->989357155],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mat", ".", "sol1"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->159825364],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "4.911764705882294`", ",", "5.911764705882386`", ",", "8.11764705882351`"}],
   "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->924913556]
}, Open  ]],

Cell[TextData[{
 "This compares the solution to that from ",
 Cell[BoxData[
  ButtonBox["PseudoInverse",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PseudoInverse"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1867744758],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PseudoInverse", "[", "mat", "]"}], ".", "vec"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->1417679507],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4.411764705882339`"}], ",", "4.66176470588234`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->137724080]
}, Open  ]],

Cell[TextData[{
 "This technique is fast, but is less accurate because it computes ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["A", "T"], ".", "A"}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1388722621],

Cell["\<\
This computes 1-, 2-, and \[Infinity]-norms for the solution that was found.\
\>", "Text",
 CellID->1408815551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Norm", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"mat", ".", "sol1"}], "-", "vec"}], ",", "#"}], "]"}], "&"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1955757482],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.2941176470588296`", ",", "0.17149858514250854`", ",", 
   "0.11764705882350945`"}], "}"}]], "Output",
 ImageSize->{209, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->822716328]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Least Squares QR", "Subsection",
 CellTags->{"t:198", "c:156"},
 CellID->701643607],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]]],
 " is full rank, a more accurate way to solve the problem is to use the ",
 ButtonBox["QR decomposition",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixComputations#1662286926"],
 " as follows."
}], "Text",
 CellID->691397172],

Cell[TextData[{
 "Solve A . x \[LongEqual] B\n\tfind Q,R such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{"A", "\[LongEqual]", 
    RowBox[{
     SuperscriptBox["Q", "T"], "R"}]}], TraditionalForm]]],
 "  (QR decomposition)\n\tthen solve R.x \[LongEqual] Q.B"
}], "NumberedEquation",
 CellID->278255887],

Cell[TextData[{
 "A ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " program to find the solution is as follows."
}], "Text",
 CellID->1574793290],

Cell[BoxData[
 RowBox[{
  RowBox[{"L2QRSolve", "[", 
   RowBox[{"a_", ",", "b_"}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", " ", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"Q", ",", "R", ",", "m", ",", "n"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"m", ",", "n"}], "}"}], "=", 
      RowBox[{"Dimensions", "[", "a", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"Q", ",", "R"}], "}"}], "=", 
      RowBox[{"QRDecomposition", "[", "a", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"LinearSolve", "[", 
      RowBox[{"R", ",", 
       RowBox[{"Q", ".", "b"}]}], "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1962359402],

Cell["\<\
The solution for this problem is very similar to the Cholesky solution.\
\>", "Text",
 CellID->320618788],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.", ",", "2."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.", ",", "6."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4.5", ",", "6."}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"{", 
    RowBox[{"5.", ",", "6.", ",", "8."}], "}"}]}], ";"}], "\n", 
 RowBox[{"sol2", "=", 
  RowBox[{"L2QRSolve", "[", 
   RowBox[{"mat", ",", "vec"}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->603499593],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4.411764705882339`"}], ",", "4.66176470588234`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->108077978]
}, Open  ]],

Cell["\<\
The QR decomposition solution can be more expensive than the Cholesky \
approach, but it is more accurate. An example that demonstrates the \
difference in accuracy is shown below.\
\>", "Text",
 CellID->1306789010],

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "4.1799231"}], ",", 
       RowBox[{"-", "5.010860245276564"}], ",", 
       RowBox[{"-", "5.841797048032377"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
      "4.985838406555408", ",", "5.8047054724005", ",", "6.623572538245592"}],
       "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
      "3.6958074724742653", ",", "3.687168314818264", ",", 
       "3.678529157162263"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "5.488825789918355"}], ",", 
       RowBox[{"-", "7.024200075119946"}], ",", 
       RowBox[{"-", "8.559574360321538"}]}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"{", 
    RowBox[{"1.", ",", "4.", ",", "2.", ",", "1."}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sol1", "=", 
   RowBox[{"L2CholeskySolve", "[", 
    RowBox[{"mat", ",", "vec"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sol2", "=", 
   RowBox[{"L2QRSolve", "[", 
    RowBox[{"mat", ",", "vec"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->887633201],

Cell["\<\
This shows how close the Cholesky solution is using the 2-norm.\
\>", "Text",
 CellID->2029797145],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{
   RowBox[{"mat", ".", "sol1"}], "-", "vec"}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->2022883420],

Cell[BoxData["5.079531702690285`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->604937814]
}, Open  ]],

Cell["\<\
This shows how close the QR solution is using the 2-norm. It can be seen that \
the QR solution is a better approximation.\
\>", "Text",
 CellID->1585451910],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Norm", "[", 
  RowBox[{
   RowBox[{"mat", ".", "sol2"}], "-", "vec"}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->713159394],

Cell[BoxData["2.719087531641082`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->480493485]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Minimization of 1 and Infinity Norms", "Section",
 CellTags->{"t:193", "c:157"},
 CellID->601898483],

Cell[TextData[{
 "Many techniques for finding approximate solutions for the matrix equation \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"A", ".", "X"}], "\[LongEqual]", "B"}], TraditionalForm]]],
 " when it is overdetermined (i.e., when ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", " ", ">", " ", "n"}], TraditionalForm]]],
 ") work by minimizing the 2-norm. This is because of certain advantages that \
make it computationally tractable.",
 " ",
 "One reason is that the function"
}], "Text",
 CellID->1459826411],

Cell[TextData[{
 "\[Phi](x) \[LongEqual] ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["1", "2"], "\[DoubleVerticalBar]", 
    RowBox[{
     RowBox[{"A", " ", "x"}], "-", "b"}], 
    SubsuperscriptBox["\[DoubleVerticalBar]", "2", "2"]}], TraditionalForm]]]
}], "NumberedEquation",
 CellID->301822702],

Cell["\<\
is differentiable and differentiation is a linear operation. Thus, a linear \
system can be formed that finds the minimizing solutions. Another reason is \
that the 2-norm is preserved under orthogonal transformations. This means \
that a range of equivalent problems can be formed, which may be easier to \
solve.\
\>", "Text",
 CellID->377778300],

Cell["\<\
However, there are other solutions that can be found by minimizing other \
norms, such as the 1-norm or the \[Infinity]-norm. These may be more \
desirable in the particular context because they may find solutions that \
maintain important properties relevant to the individual problem. In this \
example techniques are shown to find approximate solutions that minimize \
these norms; both will use a method to find minimum values of constrained \
linear problems; typically this is known as linear programming.\
\>", "Text",
 CellID->1358227568],

Cell[TextData[{
 "In ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", linear programming is provided by the function ",
 Cell[BoxData[
  ButtonBox["LinearProgramming",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearProgramming"]], "InlineFormula"],
 ". This can solve the linear programming problem for the different types of \
numbers that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " supports: integers and rational numbers, as well as machine-precision \
approximate real and arbitrary-precision approximate real numbers. In \
addition, it provides techniques that are suitable for minimizing extremely \
large systems by means of an interior point method."
}], "Text",
 CellID->2127051413],

Cell["\<\
The solutions given in this section are suitable for dense matrix input. It \
would be straightforward to modify them for sparse matrix input; this would \
be necessary to take full advantage of the interior point linear programming \
technique.\
\>", "Text",
 CellID->192826682],

Cell["\<\
Note that the techniques in this section could be extended to add other \
constraints on the solution.\
\>", "Text",
 CellID->285221051],

Cell[CellGroupData[{

Cell["One-Norm Minimization", "Subsection",
 CellTags->{"t:213", "c:158"},
 CellID->526704053],

Cell[TextData[{
 "Minimizing the 1-norm involves finding the value of ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]]],
 " that minimizes the following."
}], "Text",
 CellID->387425837],

Cell[BoxData[Cell[TextData[Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     RowBox[{"Min", " ", "\[DoubleVerticalBar]", 
      RowBox[{
       RowBox[{"A", " ", "x"}], "-", "b"}], 
      SubscriptBox["\[DoubleVerticalBar]", 
       RowBox[{"  ", "1"}]], "\[LongEqual]", " ", "Min"}],
     TraditionalForm], 
    RowBox[{
     UnderscriptBox["\[Sum]", "i"], 
     SubscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{"A", " ", "x"}], "-", "b"}], ")"}], "i"]}]}], 
   TraditionalForm]]]]]], "NumberedEquation",
 CellID->295624755],

Cell[TextData[{
 "This is done by forming new variables ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]]],
 " and finding the minimum."
}], "Text",
 CellID->1435581774],

Cell[BoxData[{Cell[TextData[Cell[BoxData[
  FormBox[
   RowBox[{"Min", "  ", 
    SuperscriptBox["e", "T"], "z"}], TraditionalForm]]]]], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"subject", " ", "to", " ", 
    FormBox["z",
     TraditionalForm]}], ">", 
   RowBox[{
    RowBox[{"A", " ", "x"}], "-", "b"}]}], ",", 
  RowBox[{"z", ">", 
   RowBox[{"-", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"A", " ", "x"}], "-", "b"}], ")"}]}]}]}]}], "NumberedEquation",
 CellID->844101573],

Cell["This is implemented with the following program.", "Text",
 CellID->538165329],

Cell[BoxData[
 RowBox[{
  RowBox[{"L1Solve", "[", 
   RowBox[{"A_", ",", "b_"}], "]"}], ":=", " ", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "m", ",", "n", ",", "Aall", ",", "ball", ",", "c", ",", "x", ",", "bds", 
      ",", "id"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"m", ",", "n"}], "}"}], "=", 
      RowBox[{"Dimensions", "[", "A", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"id", "=", 
      RowBox[{"IdentityMatrix", "[", "m", "]"}]}], ";", "\[IndentingNewLine]",
      
     RowBox[{"Aall", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"Join", "[", 
          RowBox[{
           RowBox[{"Transpose", "[", "id", "]"}], ",", 
           RowBox[{"Transpose", "[", 
            RowBox[{"-", "A"}], "]"}]}], "]"}], "]"}], ",", 
        RowBox[{"Transpose", "[", 
         RowBox[{"Join", "[", 
          RowBox[{
           RowBox[{"Transpose", "[", "id", "]"}], ",", 
           RowBox[{"Transpose", "[", "A", "]"}]}], "]"}], "]"}]}], "]"}]}], 
     ";", "\[IndentingNewLine]", 
     RowBox[{"ball", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"-", "b"}], ",", "b"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"c", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"Table", "[", 
         RowBox[{"1", ",", 
          RowBox[{"{", "m", "}"}]}], "]"}], ",", 
        RowBox[{"Table", "[", 
         RowBox[{"0", ",", 
          RowBox[{"{", "n", "}"}]}], "]"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"bds", " ", "=", " ", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"m", "+", "n"}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"x", "=", 
      RowBox[{"LinearProgramming", "[", 
       RowBox[{"c", ",", "Aall", ",", "ball", ",", "bds"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Take", "[", 
      RowBox[{"x", ",", 
       RowBox[{"-", "n"}]}], "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1891046986],

Cell["This finds the solution.", "Text",
 CellID->68718102],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.", ",", "2."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.", ",", "6."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4.5", ",", "6."}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec", " ", "=", 
   RowBox[{"{", 
    RowBox[{"5.", ",", "6.", ",", "8."}], "}"}]}], ";"}], "\n", 
 RowBox[{"sol3", "=", 
  RowBox[{"L1Solve", "[", 
   RowBox[{"mat", ",", "vec"}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->1746421875],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4.5`"}], ",", "4.75`"}], "}"}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->334768554]
}, Open  ]],

Cell["\<\
Here 1-, 2-, and \[Infinity]-norms for the solution that was found are \
computed.\
\>", "Text",
 CellID->1293517785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Norm", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"mat", ".", "sol3"}], "-", "vec"}], ",", "#"}], "]"}], "&"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1176453914],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.25`", ",", "0.25`", ",", "0.25`"}], "}"}]], "Output",
 ImageSize->{125, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->390691631]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Infinity-Norm Minimization", "Subsection",
 CellTags->{"t:214", "c:159"},
 CellID->1503089771],

Cell[TextData[{
 "Minimizing the \[Infinity]-norm is similar to minimizing the 1-norm. It \
involves finding the value of ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]]],
 " that minimizes the following."
}], "Text",
 CellID->1579641536],

Cell[BoxData[Cell[TextData[Cell[BoxData[
  FormBox[
   RowBox[{"Min", " ", "\[DoubleVerticalBar]", 
    RowBox[{
     RowBox[{"A", " ", "x"}], "-", "b"}], 
    SubscriptBox["\[DoubleVerticalBar]", 
     RowBox[{"  ", "\[Infinity]"}]]}], 
   TraditionalForm]]]]]], "NumberedEquation",
 CellID->1842451338],

Cell[TextData[{
 "This is done by forming new variables ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]]],
 " and finding the minimum."
}], "Text",
 CellID->1512931673],

Cell[BoxData[{Cell[TextData[Cell[BoxData[
  FormBox[
   RowBox[{"Min", "  ", "z"}], TraditionalForm]]]]], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"subject", " ", "to", " ", 
    FormBox["z",
     TraditionalForm], " ", "e"}], "\[GreaterEqual]", 
   RowBox[{
    RowBox[{"A", " ", "x"}], "-", "b"}]}], ",", 
  RowBox[{
   RowBox[{"z", " ", "e"}], "\[GreaterEqual]", 
   RowBox[{"-", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"A", " ", "x"}], "-", "b"}], ")"}]}]}]}]}], "NumberedEquation",
 CellID->1636247631],

Cell["This is implemented with the following program.", "Text",
 CellID->1885834382],

Cell[BoxData[
 RowBox[{
  RowBox[{"LInfinitySolve", "[", 
   RowBox[{"A_", ",", "b_"}], "]"}], ":=", " ", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
     "m", ",", "n", ",", "Aall", ",", "ball", ",", "c", ",", "x", ",", "bds", 
      ",", "onesT"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"m", ",", "n"}], "}"}], "=", 
      RowBox[{"Dimensions", "[", "A", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"onesT", "=", 
      RowBox[{"{", 
       RowBox[{"Table", "[", 
        RowBox[{"1", ",", 
         RowBox[{"{", "m", "}"}]}], "]"}], "}"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Aall", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"Transpose", "[", 
         RowBox[{"Join", "[", 
          RowBox[{"onesT", ",", 
           RowBox[{"Transpose", "[", 
            RowBox[{"-", "A"}], "]"}]}], "]"}], "]"}], ",", 
        RowBox[{"Transpose", "[", 
         RowBox[{"Join", "[", 
          RowBox[{"onesT", ",", 
           RowBox[{"Transpose", "[", "A", "]"}]}], "]"}], "]"}]}], "]"}]}], 
     ";", "\[IndentingNewLine]", 
     RowBox[{"ball", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"-", "b"}], ",", "b"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"c", "=", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", 
        RowBox[{"Table", "[", 
         RowBox[{"0", ",", 
          RowBox[{"{", "n", "}"}]}], "]"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"bds", " ", "=", " ", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"n", "+", "1"}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"x", "=", 
      RowBox[{"LinearProgramming", "[", 
       RowBox[{"c", ",", "Aall", ",", "ball", ",", "bds"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Take", "[", 
      RowBox[{"x", ",", 
       RowBox[{"-", "n"}]}], "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->507312201],

Cell["This finds the solution.", "Text",
 CellID->886083617],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.", ",", "2."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"5.", ",", "6."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4.5", ",", "6."}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec", " ", "=", 
   RowBox[{"{", 
    RowBox[{"5.", ",", "6.", ",", "8."}], "}"}]}], ";"}], "\n", 
 RowBox[{"sol4", "=", 
  RowBox[{"LInfinitySolve", "[", 
   RowBox[{"mat", ",", "vec"}], "]"}]}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->1755742621],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "4.3999999999999995`"}], ",", "4.6499999999999995`"}], 
  "}"}]], "Output",
 ImageSize->{86, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->335256515]
}, Open  ]],

Cell["\<\
Here, 1-, 2-, and \[Infinity]-norms for the solution that was found are \
computed.\
\>", "Text",
 CellID->1965002129],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Norm", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"mat", ".", "sol4"}], "-", "vec"}], ",", "#"}], "]"}], "&"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1787258537],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.3000000000000007`", ",", "0.17320508075688815`", ",", 
   "0.10000000000000053`"}], "}"}]], "Output",
 ImageSize->{139, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->189879261]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Finite Difference Solutions", "Section",
 CellTags->{"t:115", "c:160"},
 CellID->1711482715],

Cell["\<\
One way to solve partial differential equations is to form a spatial \
discretization and solve them with finite differences. This example considers \
a basic finite difference solution for the Poisson equation on a unit square.\
\
\>", "Text",
 CellID->1494023087],

Cell[BoxData[{
 FormBox[
  RowBox[{"\[CapitalDelta]U", "\[LongEqual]", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], ")"}]}]]}], TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{"U", "[", 
    RowBox[{"0", ",", "y"}], "]"}], "\[LongEqual]", 
   RowBox[{"U", "[", 
    RowBox[{"x", ",", "0"}], "]"}], "\[LongEqual]", 
   RowBox[{"U", "[", 
    RowBox[{"1", ",", "y"}], "]"}], "\[LongEqual]", 
   RowBox[{"U", "[", 
    RowBox[{"x", ",", "1"}], "]"}], "\[LongEqual]", "0"}], 
  TraditionalForm], "\n", 
 FormBox[
  RowBox[{
   RowBox[{"0", "\[LessEqual]", "x", "\[LessEqual]", "1"}], ",", " ", 
   RowBox[{"0", "\[LessEqual]", "y", "\[LessEqual]", "1"}]}], 
  TraditionalForm]}], "NumberedEquation",
 GridBoxOptions->{
 GridBoxAlignment->{
  "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
   "RowsIndexed" -> {}}},
 CellID->1667222485],

Cell[TextData[{
 "This sets up the problem for an 80\[Cross]80 grid with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "i"], " ", "=", " ", 
    RowBox[{"i", " ", "h"}]}], TraditionalForm]]],
 ". An abbreviated form of the variable matrix is printed out."
}], "Text",
 CellID->641679339],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"num", "=", "80"}], ";", 
  RowBox[{"hStep", "=", 
   RowBox[{"N", "[", 
    RowBox[{"1", "/", 
     RowBox[{"(", 
      RowBox[{"num", "+", "1"}], ")"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"u", "[", 
      RowBox[{"i", ",", "j"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "num"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"j", ",", "num"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Short", "[", 
  RowBox[{"vars", ",", "6"}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->41775339],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXNtu3DYQlVfyrpMmjuMGubRB64cURduXtvmEAulTf6DJy8IIkDwEKdJ8
evusitSOzT0a7fAyo9W6XsD2kiJnOMPD2xnKf6w/v3v7Yf35/eX64vdP67/e
vb/8++LVx09dVn1UVUf/VFX129Oq+9523/yP+/zb/fRf+29Pu9+L7uEXLlVT
qbZ6vMk/hvwz4/IPjeU3kP4V0ijnu5j63oth7t2NdsyvN/kPjcufG8lH66n+
y5Fe+F6o/4v3HvWhF1f1nxpKLrpfx10SJZwwLaoj+n2lqBOfS2lN3bH2nijq
XGx0Lg3tXG10NJH2Yfoeo2dXmvzItW+XL+5E+CpMO1ultkj46kfNA5TCeJFL
u3LY6gU1raVPNSLN1V4pao6tp6kzzdpBjxZoRhQj2jSsXbHWxdvMyXTrQb2R
wo1GXnoNtVaBNPTKLumcnMjxkWzx2aglu0bs/XxdwuqqpCXZolMVi1icFFvE
14pDOLbIyVps/HE1YkVvNVCL68nwOc4hS3guraQ/t7eIngeindSGqTc9krmW
hrPkctxLTZgb1vkSZM5uZt5qecO0aPa4BSxxXi1A6kTj/LQNdsvJtRFl3F6c
85NkyyNIS3vx1HTcvtz5ZrEZEzH7Nn/2vsZ1NVwt7uVI3PEc7cqSv9XiGJTZ
aki1QXPm/VZs7XNIIz+DaR5ZEhvD24RtwdnmsaGnLGU/OdB2Y/rrBFnfQN18
HNEYcX/dekwcDM2zxP5x5YizkcrR7oDyn0WWQ3ln7TYXRPmI5Ocg7wKeE8+I
6w/P6bqvyJanf/u/8etj5Z8oyf/xyqth7k3l26Xyz5Tk/+S9WsLDE19LY8+S
G54D5z9Wzpp3j9F5aL7lWFJuvz8HO3/wqkq4d56fnYIZPjzWX5N735+102lG
Zh1nYTw17d/afj3PY4T4Xbhk85Scow5PJM2DeswmzsL6tthEBIAv6D5Lpnem
saCEqx8/wetxzjcfyZKW2SN5T71jjeymimWMb5HNa8HnNjMcnQp1VgA8i3r/
b7GlNuNzLLrSoziVpb/PPRfbaM+sl8YGuB0h2j272IOhfGRyQnlqPTzADe6/
e4xKfP8ppOPm1vnx/ZLPDpXpl2TLa4Jei6Uo0DXiQubdobWuZIY+lskPy9EK
UxIZSIkgHAf6YiIXTVBuiojAg8AnuK5plD+UiADeyE/VV195OcwlBhzXD4kx
Ty1/KBGCl5l6/GyjEimwZJSXrX0UYtluzz2WN/BJ1xQRgCl1rRJ9yN7qH9xm
o96nFrU0K0BvaFpBq0+MFb7VBjGEKRjl+cUlNGMESf7fi7WafZmmGd8B4E8b
fsYdvV9Ko2yf7zfQ/k+Xo9U56Uin8zypg5kmSkoeY4W6dLwpaZFONzqMkhRZ
Ik6J9kqDaFw1PIHkebnkVOrnnxli33lCL8px8zFvoyUV8/jWgZ+dhwzm6IrA
c/ZzxT/de9eZXyxH083Hv3RjRu9NEI5lYN9hGKC82c4ljO4YTzw/HONRmjs5
fPaYz4p5qMc4bKTm4T01upGK9FIrpNk3VX7q7I7vLYS4cv78KlF/XhStR2/e
2w9YSidCIUnJQ6MU50hFn00rJansvSJRqoS0F4LWvFjYNbIwWlAH6ZzntGaE
5TDKEZYL/8bK46I1XDn31kgTyH/B1HPlMYriGXfvqvKoh4RV4p2njoqMlR+L
ili350xJ/mvvdelWCkqbKkoyVn4sSmLdHtSbK/9NlNepdAk7L40njQiApY7Y
KFBJ1OROa/8ewRTRrJOJdKTsLf5sY5A+OO1EoIpWnxhu3oZL3r+mkpgGzukW
d/6n89vJZJpi8davrNK9ubz/jCDt8POZ1bG9db5UKVaRz2zr3d23iafEWWZz
r3jt5UTNuzNBn41UO/QdVmzvFouaFuUjnDuD5WHQ1zK4H68bu6BonU2kNQFp
W5y7JperfbNeW14Jd831SWr7pF1FqTzkprX9Fzd74W46FUXT8M1xUlL45ZxW
SIhI5aN1/lMRlur7XUIPMT2060eutpQzRi74vN2+F8LVb3Y8P2/j/ncPPb/0
XnCmxrC31dF/okGbEg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{500, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//Short=",
 CellID->130715566]
}, Open  ]],

Cell["\<\
This is the finite difference formula for the differential operator.\
\>", "Text",
 CellID->1483891766],

Cell[BoxData[
 RowBox[{
  RowBox[{"kern", "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", 
        RowBox[{"-", "4"}], ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "}"}], "/", 
    RowBox[{"hStep", "^", "2"}]}]}], ";"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1041318551],

Cell["\<\
In this step the application of the finite difference formula to the variable \
matrix is made. The boundary conditions are incorporated by allowing an \
overhang of one variable with a coefficient of zero around the boundary.\
\>", "Text",
 CellID->1596647254],

Cell[BoxData[
 RowBox[{
  RowBox[{"lap", "=", 
   RowBox[{"ListCorrelate", "[", 
    RowBox[{"kern", ",", "vars", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "2"}], "}"}], ",", "0"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->2042228713],

Cell["Here equations for the solutions on the grid are formed. ", "Text",
 CellID->506849390],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Short", "[", 
  RowBox[{
   RowBox[{"eqns", "=", 
    RowBox[{"Thread", "[", 
     RowBox[{"Map", "[", 
      RowBox[{"Flatten", ",", 
       RowBox[{"lap", "\[Equal]", 
        RowBox[{"Table", "[", 
         RowBox[{
          RowBox[{"Exp", "[", 
           RowBox[{"-", 
            RowBox[{"(", 
             RowBox[{
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"hStep", " ", "i"}], ")"}], "2"], "+", 
              SuperscriptBox[
               RowBox[{"(", 
                RowBox[{"hStep", " ", "j"}], ")"}], "2"]}], ")"}]}], "]"}], 
          ",", 
          RowBox[{"{", 
           RowBox[{"i", ",", "num"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"j", ",", "num"}], "}"}]}], "]"}]}]}], "]"}], "]"}]}], ",",
    "20"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->1697553728],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{538, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]//Short=",
 CellID->276418174]
}, Open  ]],

Cell[TextData[{
 "The approximate solution to the Poisson equation will be found by solving \
these equations. One way to solve the equations is to use ",
 Cell[BoxData[
  ButtonBox["Solve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Solve"]], "InlineFormula"],
 ", which will recognize the equations as linear and sparse and accordingly \
use a sparse matrix solver.",
 " ",
 "Another method that also gives you the possibility of solving additional \
Poisson problems is to construct a matrix from the equations. The matrix \
represents the linear operator when the grid is flattened into a vector."
}], "Text",
 CellID->1364309358],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"vec", ",", "mat"}], "}"}], "=", 
  RowBox[{"CoefficientArrays", "[", 
   RowBox[{"eqns", ",", " ", 
    RowBox[{"Flatten", "[", "vars", "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1264480887],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   TagBox[
    RowBox[{"SparseArray", "[", 
     RowBox[{
      InterpretationBox[
       RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "6400", 
        "\[InvisibleSpace]", "\<\">\"\>"}],
       SequenceForm["<", 6400, ">"],
       Editable->False], ",", 
      RowBox[{"{", "6400", "}"}]}], "]"}],
    False,
    Editable->False], ",", 
   TagBox[
    RowBox[{"SparseArray", "[", 
     RowBox[{
      InterpretationBox[
       RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "31680", 
        "\[InvisibleSpace]", "\<\">\"\>"}],
       SequenceForm["<", 31680, ">"],
       Editable->False], ",", 
      RowBox[{"{", 
       RowBox[{"6400", ",", "6400"}], "}"}]}], "]"}],
    False,
    Editable->False]}], "}"}]], "Output",
 ImageSize->{450, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->31349214]
}, Open  ]],

Cell[TextData[{
 "Now the solution is quickly found using ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->54307570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"sol", "=", 
    RowBox[{"LinearSolve", "[", 
     RowBox[{"mat", ",", 
      RowBox[{"-", "vec"}]}], "]"}]}], ";"}], "//", "Timing"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->66779015],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.10900000000000001`", ",", "Null"}], "}"}]], "Output",
 ImageSize->{93, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->11356225]
}, Open  ]],

Cell["\<\
This repartitions the solution to restore the original structure.\
\>", "Text",
 CellID->1991746674],

Cell[BoxData[
 RowBox[{
  RowBox[{"sol", "=", 
   RowBox[{"Partition", "[", 
    RowBox[{"sol", ",", "num"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->311260482],

Cell["This makes a contour plot of the solution.", "Text",
 CellID->84756882],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListContourPlot", "[", 
  RowBox[{"sol", ",", 
   RowBox[{"ColorFunction", "\[Rule]", "Hue"}]}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->727257813],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJx03Xd8VfX9x/GoIHFHjRI1SoghhBDjHbk3uclN7ocQCCNAwD3qXnVPRFzH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   "], {{
     {Hue[0.], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{7159, 7158, 1791}, {7275, 7276, 2125}, {7652, 7675, 
         3485}, {7704, 7729, 3643}, {7628, 7651, 3406}, {7602, 7580, 3224}, {
         7192, 7191, 1869}, {7676, 7703, 3564}, {7752, 7726, 3709}, {7325, 
         7326, 2287}, {7783, 7755, 3792}, {7197, 7198, 1881}, {7701, 7674, 
         3547}, {7725, 7702, 3628}, {7252, 7253, 2044}, {7757, 7790, 3799}, {
         7728, 7758, 3721}, {7301, 7302, 2206}, {7673, 7650, 3466}, {7514, 
         7494, 2903}, {7224, 7225, 1962}, {7370, 7369, 2423}, {7581, 7604, 
         3247}, {7165, 7166, 1797}, {7626, 7603, 3305}, {7250, 7249, 2026}, {
         7392, 7393, 2528}, {7274, 7273, 2105}, {7536, 7557, 3088}, {7222, 
         7221, 1947}, {7324, 7323, 2264}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJw1mXeglcURxS+C4d7v3ncxWIi8oiQWlN4EqYIFHlWKogiY2CjSYjQGEFBA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         "],
         VertexColors->None], 
        PolygonBox[{{7674, 7673, 3466, 3467, 3547}, {7675, 7676, 3564, 3484, 
         3485}, {7225, 7251, 2043, 2042, 1962}, {7515, 7514, 2903, 2904, 
         2984}, {7198, 7224, 1962, 1961, 1881}, {7371, 7392, 2528, 2527, 
         2447}, {7703, 7704, 3643, 3563, 3564}, {7276, 7301, 2206, 2205, 
         2125}, {7193, 7159, 1791, 1871, 1870}, {7249, 7222, 1947, 2027, 
         2026}, {7755, 7754, 3711, 3712, 3792}, {7756, 7757, 3799, 3719, 
         3720}, {7727, 7728, 3721, 3641, 3642}, {7627, 7626, 3305, 3306, 
         3386}, {7557, 7558, 3167, 3087, 3088}, {7302, 7325, 2287, 2286, 
         2206}, {7223, 7192, 1869, 1949, 1948}, {7726, 7725, 3628, 3629, 
         3709}, {7166, 7194, 1878, 1877, 1797}, {7651, 7652, 3485, 3405, 
         3406}, {7323, 7300, 2185, 2265, 2264}, {7253, 7275, 2125, 2124, 
         2044}, {7702, 7701, 3547, 3548, 3628}, {7603, 7602, 3224, 3225, 
         3305}, {7273, 7250, 2026, 2106, 2105}, {7604, 7605, 3326, 3246, 
         3247}, {7369, 7346, 2344, 2424, 2423}},
         VertexColors->None]}]}, 
     {Hue[0.10526861255814486`], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{7517, 7537, 3014}, {7757, 7756, 3800}, {7704, 7703, 
         3644}, {7222, 7249, 1946}, {7250, 7273, 2025}, {7192, 7223, 1868}, {
         7226, 7227, 1970}, {7792, 7816, 3888}, {7916, 7890, 4189}, {7699, 
         7672, 3541}, {7676, 7675, 3565}, {7725, 7726, 3708}, {7301, 7276, 
         2126}, {7194, 7166, 1798}, {7626, 7627, 3385}, {7605, 7604, 3327}, {
         7750, 7724, 3702}, {7157, 7156, 1782}, {7701, 7702, 3627}, {7346, 
         7369, 2343}, {7754, 7755, 3791}, {7106, 7107, 1645}, {7248, 7247, 
         2020}, {7840, 7866, 4046}, {7300, 7323, 2184}, {7817, 7839, 3967}, {
         7132, 7133, 1727}, {7251, 7225, 1963}, {7103, 7102, 1624}, {7325, 
         7302, 2207}, {7892, 7925, 4202}, {7414, 7415, 2614}, {7327, 7328, 
         2293}, {7074, 7073, 1546}, {7322, 7321, 2259}, {7606, 7629, 3333}, {
         7728, 7727, 3722}, {7922, 7952, 4278}, {7558, 7557, 3168}, {7653, 
         7677, 3492}, {7652, 7651, 3486}, {7130, 7129, 1703}, {7760, 7791, 
         3809}, {7254, 7255, 2051}, {7705, 7730, 3651}, {7731, 7759, 3730}, {
         7578, 7555, 3139}, {7392, 7371, 2448}, {7865, 7893, 4124}, {7947, 
         7920, 4273}, {7079, 7080, 1563}, {7199, 7200, 1889}, {7602, 7603, 
         3304}, {7781, 7751, 3783}, {7862, 7838, 4026}, {7167, 7168, 1808}, {
         7888, 7863, 4107}, {7045, 7046, 1480}, {7514, 7515, 2983}, {7277, 
         7278, 2132}, {7224, 7198, 1882}, {7034, 7033, 1469}, {7190, 7189, 
         1861}, {7275, 7253, 2045}, {7159, 7193, 1790}, {7673, 7674, 3546}, {
         7837, 7815, 3945}, {7814, 7782, 3864}, {7648, 7625, 3380}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJw1WXd01tUZ/iHk94VvQwYk30iZYWgtMntOtbKHHmXZU2UlJGETkBAQAS0Q
hnUUUUatRZZQmYEEpEiYCRtayxJFKMjqORWBhKUCfZ7zPv3jObnfve/vve99
972pN3hM7/zHHMe5Ws1xauDv4LDj5AAlwCZgSdRxSoHONR3ndqLjzMB4GNAK
4yJgSsRxpgK7EhxnJ/Ax1jYBmaA/jfX+McfpCrzqc5zdPuOdC5RqD377BrA7
wXi8BJrfASuBT31Gmwds1jd9MNcXWAEsB14G784xm+fv1tizDTALmJlotC+J
F7+h3K21Rvl9+DYDOOXFN8DkiJ2JZ9kBHMBZNgINcJ4CYJ3rOOuBh8ADYCvw
d6Cmx3ESgSXguRS4CdzQeBlwS3NF2vskcAL4Pu44/wUG4GwDgR8wvg4MwjgL
WOvantzrZ9fW+A1pSTNTZz0lnrexVgUMx9qIsI0rgWFhm/tcMlNWj8fsMg7Y
I/tshC7+Anzod5xLQEOcuRFQCIwH2mKPdsAcYLZ03VZjykE9NRQt9XUYvC4C
bwUc59fAFtdk4N4ucD7oOP8GfhVynCdDxoO8TutM20D7BeADrRfIDpuP0jc3
An+AraYB5bDV3gTzpTeBPQnmUwfB+xCQDt5pQG/I0AdYDiwDZkO2xcBmnDUP
2AuZy4Fk+EIS0A9rY4C70M0qn61VACleoynC2sfAQnzbFegNmfsAy2V3jvsC
KzTHvbYACQHbk7z2Aale49kNawOAfdhrAvAB1j4ELgHfAefhi1VRo+d3NUEb
I0/8HgLcjZpP05fbeM23WwGNfebjHHMP8ubcUxh3iVnMuT7jTZ7kxT0Y+0eA
+TUtB1AWji9LpnNBsyFt98uQ0V6M2jrp3gfmAReBC8ChqJ2Bsk8A8jGeHTX9
UE+0HW1KW9KG42JmI9omhYDNU4EhygvJYZvLU15J0lyu8lgakK5YoP/XFk2O
/KgOUDds8TEUWOA3W14FrvhNt9QRdUMdj4CsRVHzf8ZBqngMlUzz/caD3172
G+807c09GEuNdXbGFONsvmgZb8nygzwg12u+Rx+jb9EHZ0A3RcB+6GYfMD1i
cxxXJFgscI5jxkQlZD0XNb+mP/PbmcCBBOPRF2t9gDWIrbVAr0Tz4WXKXXk6
1xbl4J6JRsM15roXE22O40+Aw/CFI0AUvhAJ2Z7JOkuO1/gyDiqVE18FzThg
D7Bb4wJgr+bGhoyGa7uAfGAMsBPYARQCE4B9QAVQSzambZkryGs8UC6epJ0I
7Nc3T8hv3wCmirZQvMq1Rt9+UzT89jXggHhwPAk4qLnHxZO8pgA/Q9c/ASXI
YaXASMyNArYDXwAZwC90Tp4rrrkxOusB8U6Tn+7XXF35HfmUAQGM/fTzkO1B
3tvEu550yD2aAs2A1yU3ZRktHuSVLruRxwit5UvXZeJLOu7lAx5E7Iw822ag
uXTAs08OmR8zlzB2NnttT+59SOcKgkdItYw1rZPyzHzlmjT4Zp2o1T/W3BjG
0ajVL9bcZ0HTXnlmbk1bi0StvpAmithJjJnvlcgH6Yul+s3YYI1nrWKMtMW4
NfAcvn3eY7R5kp3fFKi2lYv+Hs5+B9gA2YqBw/i2JGpyzZVsHZQDKSPtFNRZ
WfOnyq+OAUelt6kaH5EOp2h8WLqbrPEh2bK55riWJhvSdsPlR2MVO4yZ+xGT
mbJuBB5DDFZXnWLcU3fpUavN1GHHmmYT2uIDnaWjxjxTtUTj0Vt5gb5WP2Rx
TJ+Lyafz5UtRzY2W7xUqF+5TXW2tusX6NROoSrS+s7PkOIle5jgQge6i1GUG
dJphvST7wzDGoQzLVcxZHJOmVPnrGHzhMrAC9bcv0Mpre3KvIq/ZbnHUejfa
8CFo07BfR+S0StB/ifE/4+b7jIGn8E1LYAYwHdiGta1xsx9tStn8GdavUMYU
1dUh8qs1rvV47O1+cq3Gs662Et+Lqvscc69Lqr2UmTTL1VtUKaceV1/JHqyB
ai558NsWXutlP4oaHddvY3whanRcZ6ywZiQpVii7L8N6LZ7hlHqzTNWxk+rT
GqtPPKE+tJH6wBvqexkH1Cd70Ztxq5esh5WSvZP8qoXOSV1OAxpAhw2VlwuU
s5kD66gPyMS4ifLwRPk5awXrQDhsY9YR1gTO7RRNWHmHvtpAvMcpphhbEfUN
BzWXrjo+XnW7Qjngx4jFFGNpk2vfMlbpm+QRkc+PUk7doTgMKQ+UaY5j5oYm
ytGTdKZ5irXvZMewzpGt+8FN1dEOyjUt5RdF8slb8ouOil3qgnUwSXWSY9a5
ZPVFjTBuHLI6WahxpnTLuQrZIFV9F2sbdcCzs8aViyZFfRht10i8aENvzHyu
hez7N3zzGXAfuAdMitgdjHevMmC1azHC2PjRtZ6dvTtj8RZw0mt9LXvZRkB/
5KwBwDr1M39E3L4NfAN8DXwJ2n8B9UFbDzgNHl8BTbCWCczHNwuAK8Blj90t
WwDVfXbHvKR53iF5l+RduGfU5vn7hNdkoiwN2XMHbA/ybhw0PjV0hyU/yjoQ
WC+ZW2KveMzkpHyU9TjjwGcyL9W9hWe/CXRHLPUA/gHex4AfMHcDeBbj3wKT
gSnqydibTQJe152Id6PXNMfxAaBb3HiS19GgnZW6aKN7M2UdBGyQzGew19dA
M9A2DZqs1AHPTpmXBEzmm5Lrk4DNcUxZF2uO4+uyBXmSVxPJOlmyU+bvRfcM
xk8HTbarHruDUsYXsdZTe5DvDr1LhOA7QdfGfGcIuzZXpns+1wKuvRHwreA2
70JAF93rFqqvXw7fXAH4eScFlkZsjmMvMB7fFAIVvCv67B7DPjwZa0mu3Ul5
r6mNcS2gs9/2WKB7wC7dW7lGGQt1D9wnnn7JOQDoD4xgT6k78jbF0mrFCmNq
FDCa5wS2A8m6R/E+lQvUAmoDg4FsjZOAHM1xnCxazh2M2RsB7yqtoN9vg3YH
ZJ/Mvpe2uOaxuxFtslL3Zt5d7ki3KzWmjofrDNv0jrKDb0nAyxj/Xjr266z9
pGOfxq8A34D2LN9r+C3QyW86pS55t7ou32Is0GcYewl6E2IMTlQM0Pf3AyPh
+6PilrOYu0bEba6h6sQsvhkophnbRxQnMeX5J0HbIm45h7lneNx4NFBfRN6j
45ZXuQfH+XHLsZw7ozzVTHWAeets0Hpq9nnD4saTvNhrcY02eEJ991dB49FU
dXFOwN5ezii2usM2PYCPgD97TC/MFXOBP2ncHnhfc88oj3DtPcUgYzGiPtNJ
tB6wl+6Bjzw211N3w6fFg9++Kz0z17BPrRuys/KMzdXL/kZxTdp3tO9cvRGd
l2wdgHmSsZ/Hcj5zPe+wczV/Qd+8q73PSU/vied5zT30mMyU9QWeI2BvRMuU
t/qoT1yhujNTvekp5fkmynuT5EeZylscT1TOz9R4QtB8z9UbJ32wM/buoprD
2tPNYzaibRZprSuwUDTMs82VE5kbmYuzgGLl5BvKte1ly5uqFR2ku1uqnR2l
x1Z8S4qaHNyfZ2km3jwTeWcDG7VHfszeXPnWWgk8Dl9Mjts5eT7m3l7SHXPw
PdBeAGayVgfs7yzFDn8fVd2K614wTb0f6x5rIGshe4Zp+s3YmyNfZgx2lc4W
SUfkxVqYoXtHkXr7k6pLz0PWF+IWI4yVt+VnZ+WH09U3HZcMM9RLndDcO/Kp
b/XN0LjFJGORd+3lemeukn5GKgdvV44OqS/N0p23nuo632D5FpsGmnTlRObG
KuVJ5jTmNr6ls0eoJ/rdfOuMWI7nPneUa7uqrjRQH8K6VMD3d9Dk6i2C/Vp9
1ekCyZCNucHqZdnTrgTvVUBEcnEtR9+Spg1s0FY5kXat69oZKPsw5m7YJKCe
jHFaxzUarg11rS6mAHmSK9U1Gq4NoQw4wybgMfCuFjDaVK39/5s8vT2Uiu9w
1ZKtqrnJoqPctTWXo7NUB88ainHGei3RcC1bNbiWxllA64CdeZZ8OYgzhdQT
sTfKk2xbJNMQneVzzd1TXe4uv72vWt1DcV/qt3fhBMm1QH59VX3YXfHoprxA
3Qa1N3Vc7DedUVdOwHRX4rdzUofso3nX5NsG3zjuuMazq3IN39n9shVzK7+l
TJSFPLyi4dorHnsX4duOq7raTu/8bylO2TvzDYZvLex9S/SOwjrM36tcuwPc
kxz0sYh6GsZAQ/X1hfLhobLxVum0kd6xJ4iGvthOe9Mn+Y57zW9zXCvVW06C
+vAreuttK7te0lss7cz+5rLecrnGuYXKNddkE9o+rBxMHzjC/yUAj/zWi25X
7xWQXrPkS4wtvsEUK7eyhrJujlMs7tX/ZLi2yWNvLKTZoDzsqNayDxzo2v+k
1qlPGqDxWvVE7I1C8pMZMcvhzE0d/dZjsdf6THbIVW+4WXYvV9+Zoj4wR71g
qfxqveoOZWEtpSyDXHsPo0zlepdOURxW6N06VXHPN2i+RddV3H8asZyTrjzC
OGZv6NV73zrdO7jXA4/xIo86ygsF8pNy6XCQ4rZYMu3R/4mSFOfMyduVp+jX
7K2D0in1GFIOyNK5bqt37aI+k7HLmHpOfRV1yb50tXS6SLX8P/KZfrIRbbNG
tP005jdjY/Y/U/4vjr35bvlBbfXqWerTGeOMdUd5sZfuPXt07iT18tUU971F
81f+byhmfIqVV6sr95FmsGxcorxbJh8KKtc80p7ci/6drXvCJskUlM4GSYeV
qoWsXfS3vbJNsu4W/wN18QoL
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1FlsVUUYB/AJSDd674UoUry3F3wBo3GFGqrwIoJGkCqbWmLSmmhDWVp9
wESNPmhbUB/EaAxLWyLVkrgCAi5BNEZE0mJUXFpj0iKN8IBWwBKhLf5O5uGX
zv+c+b6ZOefcXv1Iw5J1Y0II8xlHyIUwldrCEGqYwFA2hPOMFoTwPh/wlfwb
nxsfYC3n5F46jc8zRLP8FlOMyzhskQr9b2WL3vewkMlyGYuMh80b4ea8a8xJ
h/AyL7FOvpdfja/JhDCDvLpyitWWUM2oNUfYq88+Clz7Xb6UjWfKcND1yXoN
l4dwOhXC3HRca5Je/2bj/heYN5+d8pescW01WbrVHWWZ2ssYx3G1Q8n5SpyT
geIQbrHGTF7V+w7m0WZOK+1FIWxnkLvMWcB695/kG0b0H+Ws3jvoSNaQ/6PX
uIeNtCS9kn3rc44OrtTrep7R52mu4wr5cn5S8zPTXbug16R8vH9tOvZdKNcx
zfOdyneuN8iN3CT3yf3cIN9Nt3FXOq71mPxoPu7/MGXmd3h2T3HA+095dqX8
Y84Z9k1QQx2bneF5XrD/Y/xIv/31Mcu+Kmii0pzZufiuV/IOW+Vt/KKmhQ1J
vf7HqNf7KrL02MdBHrSHnQXxe23Sv5GJpb59aqhSd18+nnNTOr6/L9S9SYma
ah5KvhF11XSOD+FtVrA/ORefOvtnjLduu7ydo/KzPMdWdVuYZ707eY1vneEI
630705nBRXNO0Obsf6Xi9/qH3EWVcXsq3tsjH6HSeDYn9dsm7y6P4z95g4/k
Vj40DuZd8rffmn28Yr1N9PO4OU8k34f7XzvXIZabs4x3PfP3GOF2eQVjPfMx
7HZtlVyfvCf5pHyKB+S1Sb1rS4vi/Ga5icXGVUXxN9HmObWyyjOqz8Tn+IP8
PaWeZYqHqVI3i9f1HuAEH5vzCTn3u9R10yI35+O3OCUT+05Td1su7nVXYfyf
t0ZenYtnGi2Me7pfXpKL+99cGP9vLU5+B7yY/F7oYYO8MR/f642Z+DsZlP+m
xl5q2cX/tRTOfw==
         "],
         VertexColors->None]}]}, 
     {Hue[0.2046832111627955], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{7254, 7227, 1971}, {7105, 7080, 1564}, {7167, 7133, 
         1728}, {7817, 7816, 3968}, {8000, 7976, 4425}, {7837, 7838, 4025}, {
         7841, 7867, 4053}, {7108, 7109, 1652}, {7578, 7579, 3219}, {7760, 
         7759, 3810}, {7731, 7730, 3731}, {7246, 7245, 2016}, {7750, 7751, 
         3782}, {7188, 7187, 1857}, {7978, 8005, 4446}, {7835, 7813, 3939}, {
         7895, 7928, 4211}, {8004, 8037, 4524}, {8059, 8032, 4590}, {7327, 
         7303, 2213}, {7414, 7394, 2534}, {7081, 7082, 1571}, {7794, 7818, 
         3894}, {7169, 7170, 1814}, {7699, 7700, 3621}, {7031, 7030, 1460}, {
         6975, 6974, 1304}, {7678, 7677, 3572}, {7131, 7107, 1646}, {7277, 
         7255, 2052}, {7654, 7679, 3497}, {7892, 7891, 4203}, {7560, 7583, 
         3178}, {7860, 7836, 4020}, {7814, 7815, 3944}, {7889, 7890, 4188}, {
         7840, 7839, 4047}, {7199, 7168, 1809}, {7299, 7321, 2179}, {7781, 
         7782, 3863}, {7886, 7861, 4101}, {7623, 7600, 3296}, {7697, 7671, 
         3537}, {7034, 7076, 1468}, {7220, 7247, 1940}, {7201, 7202, 1895}, {
         7648, 7649, 3460}, {7279, 7280, 2137}, {7792, 7791, 3889}, {7005, 
         7004, 1382}, {7077, 7046, 1481}, {7945, 7915, 4263}, {7706, 7732, 
         3656}, {7134, 7135, 1733}, {7862, 7863, 4106}, {7349, 7350, 2378}, {
         7630, 7629, 3413}, {7865, 7864, 4125}, {7779, 7749, 3778}, {7103, 
         7129, 1623}, {7157, 7189, 1781}, {7072, 7071, 1539}, {7226, 7200, 
         1890}, {7761, 7793, 3815}, {8034, 8069, 4600}, {7919, 7920, 4272}, {
         7975, 7946, 4344}, {7048, 7049, 1490}, {8029, 8002, 4507}, {7130, 
         7156, 1702}, {7538, 7537, 3094}, {6979, 6980, 1325}, {6935, 6934, 
         1226}, {7009, 7010, 1408}, {7228, 7229, 1976}, {6951, 6952, 1242}, {
         7128, 7127, 1698}, {7954, 7979, 4368}, {7927, 7953, 4289}, {7074, 
         7104, 1545}, {7922, 7921, 4279}, {7914, 7887, 4182}, {7868, 7894, 
         4132}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJxFmXeQleUVxj9wb9lddrlXEYHvFhZsWGZEMImZSQbsGhsq7GKvYIUFJIpY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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1G1ollUYAOCj7dvyfedIs717l9PIrH7osqCCco1KI5TEzcppTXNWzjkt
0qCwTM36VQqRppvGdAuDoH24aaXmKvtyDk2SPjRJoSLCnM4fYdfL6ccF932e
+z7POc95nmdMbcPDS4aGEGaSzdziEO7mGski6vi2JIROcq4KoYpZHFfzMTNz
QniP7ayQ17BQTycdvJsOYTPzEyEsoJ1eNT/Tq+cgz7JOzRa6Xd/NIj5Tc4AS
19MsoUNNezrO00F2MoQueR+PiR+lgK3yznSM85Nx3oJSMTvlrQzyk/l/pMHc
S9nHvWoq2VUYwhxqKEqF8I+6c65X5IYwhV/kQ4yPFF+dG/dz2D2/S8d15VDF
SnUvsNfzWEw9e9TsZZzrR63tGBXuOYVp7vcgm9lm/g6m5YcwlUt5IVyUX+CM
/Cy3F4Twt7mG6R2iJ/CheX9wzxPssK4qqjkvH8gQt+TEawsy58OB4Z45RdbS
KF/GbeIzxs5yv/xJRhkbyefGPjHXp8Vx7lzyaPSu1PDFMLVXqmWhvrXpeK51
iTjH+/oOFsfzX85YXlfzBmtcn5yI9x9vXzeUxvO6xFT7e8j+y3nTcz/FSS7o
u8h+NY0s40Z9E5irZzgJCuUjSIr3JWP99/qOs1w8NhnPpV/+Jx+IdzGHP+S/
p+O5Vifju1RpD/N4ybm+yC1Mt7YZnLauDWzkTvkd1Irn8xED5jrKM+bZQw/H
Mt8BJeJ6FnOP+UfxVVYI15p/NLXydRwR97Eq8wGb/3JxfI/3U+geq53FJjY4
h7eoYK2aNazQs5JDbJNvJ19ft7yH16xjKROdw8usYpJ8NJXOfyMbeMLzrM2c
kWfaShuPy+exRbyVv5iurzwde07zKz3yZg6bu9Re0/xrvb/R4l8zwHka1MxI
x55JibimJvn6dFxXXyLOcVlfMDaop5U2RsgLOSLup8wcg+qKjF0vHsdR4+Pt
fwo32fsETnje73imm+j37b3CaprlTbSId3CBanlVKp5rO1nGrpOXpuI5jMiN
Z18mv4vd4i7qGCofk4rxwtz//ymZfwuV4vM58ZudLW/gEXPPJo8l8vpUvF82
s7jVvibzdua75QGa7LeZbuf/FE+z3l7biuO3V0a/PbfKd/K1+BtSxg/pO8mr
ntFqymlS8zztarLUXMHN7nlfOv4benmOLzP/gky/+51iot6k9+Kcui5nneu9
yOE/vwrrPA==
         "],
         VertexColors->None]}]}, 
     {Hue[0.30409780976744616`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV03loz3Ecx/E3GzZhttHKmOaeu1x/uI9Q+MNKyTWb8w8RJvKflJmjKH/I
seEP5ZjGLDlzH4klbWSOP5D7D0c0/pjH949nr8/v9Xlf3/f3+8svWVO4umVE
LEAK7nWLuI+h7SMO5kUcQk1GxNLciGUoaxWxqbt4vM+M+MnL7hrxvXVEcxcF
/N6bGlFEp2Cw+CtqXEbvjhHn6AFUqznV/TSMEHOeV4tUMVtpCTLEBG02T5N5
Jorth0b1L+hfj69Z+tDJqDJPJb2ISv4SuUuRpc5XXh2GZEecUKMcT/U9xzuM
7eKviX2OAjN04g1Ejprz6XjsS87y5qFY7kTeJBzgN8nLdr4vt4QWo4lfRBfj
m3NneZ1wy/xj7KwHVraJWCd3LUaZcyfdgW3Oh+RtxiizvbPb9xgn96oaz1Bo
5/94f1HK/2BPLeRW2dVG95WoMWdmsgPkd3Cv5xv+XbkpzqnINcNaug4L0yLW
0LlId14odlPy7tUZTxchx7lUr3LktYvYwCtFNf+xPnWYYw87eDvxnJ+vXkvk
6DWWNwZZ/P50AN6a/7O8Lzgud6bYGXglvifthVXOf9z/xk0xH+kjzHZ+RhtQ
6lxmZ0cw296GyZuFPXJz9JmAB3qV0RJsMcM0sctx3jtar8YujLb/kbQw+Q/Y
W3/3BSgS89K+G9FHndtqnEQv+1yRfEdI0bcv7Yfp4o/YUyUu2dUZuh+7nce5
75N8u2LSaVv8Mf9d9e5hs5q9eWn4wT+rZzVe+KPWeqaj+OsdPaZ1qEiPeC3v
DjbIfSLvV/L9+OZP8U6jgD/CMw3Hfs9VQDsizzlLbGaCea7rcwOf9KqQdxjN
dnVM7FW8sJ9ByfMluxP/kNai0bM3iK1HhV7/AZqcob4=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNWXd41tUV/kESAiQk5AuCfr9vSCtLRpAdRgAZCjJE7VMLstHWBVplKBCm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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1FlsVUUYAODppkJQ7m1ZDLfcispSUEHFgMgWKRZCEKn6IEuEsqhBRUxB
wSdFtiKLiSYgXcREE5GlUJYoaKJUij4gRqUFpKAGRNDEJRHZgt/JPHy5578z
888/yzk9KudWPJ8bQphCAUtLQljGxMIQcshlcyaEAyy6LoSFlHIkG8I+eqZC
mE8VZfqM4kR+CG3cJWGLuJWNxkykgrT8F4z9wpjCtJgXikOYS7sbQmjPJGr1
q2GQGgaznCf0mcdU7VPokPzKP5VZ5utMF74UN3G7+Xoyn5XijzKx/j606tdP
PJqBnu/lF3VvEn/Id+LvC2KOH8T7mOC5njrKxTNYqs8S7uOkuC0T51vAAbZ1
D2ErqzqEsJo2rnQL4Ud6mW8B88m3rtt49voQ5pBhgVwv0Sj3DGayQq5qnpOn
O1kuiNP2tMdNIdzC4RvVIv92juWF0Ff+Uur1W84cY/byCX2NKyVrXAlzqTfn
u+Sqf0dBnD9XbTnFcU3NdFXfLOdTxhZneJUr9BMXlsRznc40ZslVnYl73sIy
iuTqxEh5/pTvL1bos5zXtC8uiGeQo7ZcrljTZrYwUNyNsWpdzzoqjZvOAGPu
5lVOifOK415242tz7FbbHn53n2qo5UHxVM6r9RzvUCP/TvLd0zwaO4bwkD7l
7NQ+M1kX/+nTko3vQBP7Ga1PX57Uvo2t7FbHLi65s5cZ1849VF/vZG+czUqO
M1w8gq7qbxYfZIlclVyUZ6x6xxCSu8zP2n/iTabp8wrDtQ/jD/1vNWeP4nhe
N/M0i/RZyBntv6bjOzZen4c5qX0Na6mwrkc5b383UMNQc3Zmv/mK1FjIHuMO
UWdNtYznmnt2hk3O6xIX2ZXsHdXmW8lxZoufIs9zYzru7WHxt8m7X6Q+/59l
rzo2csQ59ErFb0+1+EUG+W8ZS/nGuEO0GXOS/nKMU9s9rLamNk4w2rgZFBqT
5nNrq7Kuyck9sqbtNDDMuKHFcd+e4WOmyD+Zt9T6Nqd5Xa7FdJQrxTRaxUd5
Wa19KOW0d/Iqa+zf6uS8k/PVZ0M23rEdVCb16HOKIdrv5zfvcaM+O2jQvp0c
+WqT700m1hrc72t+P9Xns2yc76h+x9glPsTj/nssFe/1G+JVVGsfwgPJ/onP
8YH291Ox/4jkTjHKOsvScd19xCNZ73kdYzhvXLvk+27Mv6mYo0jcia7am8UH
6ZLcb76yroUsStZn7Fkmaq/gPeqMq084xwk8wiTjhmXivassiGfVSXzZXlXJ
05Qf7+Ydxt3JbPN2Scf524t7l8Ray9Oxzl7invzt+Z9UXOO8pI6kHndicMd4
vxrc5WYGuMf9k++rb+f/lrkBig==
         "],
         VertexColors->None]}]}, 
     {Hue[0.4035124083720966], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV011MkHUUx/EjksgUDYHmChVG6MSg1YW0Rbq1Et/GBXbhS860CbU2qFU2
spuKpSLUBjcN0KQuulBB1HS+0XTqUheKrQUsJjXaeKmmG7M33Po8F9+ds/P8
zu9//ud5nvwdtZU1KRGxA9OR81hEGS6lRpxeFHEKf8yL6FwQ0YWh2RF9arfw
TFbEYdr9GHgo4tuFEYMoeTjiqFonSmZENOTSoHtmxBkefcjMiJiVF7EEd/jc
0NeNB3MjWh6NaEa9gRp5NGGIf4/nF1DE/0vnn8XXZitVewmH5phDXz/yzF+l
rxqNel8Vy7Fc/intV/jMWb+o/Yqb5mwX2zBF8zOPKRTyaVDbib3q58UfUZHc
yz32ocZO3nK/TchwxwHe/agz57u0B9Clt9fz06hIjzhp9m8waP40O5iQ77KH
52g3I5P+mFobPqLpFW+ihKaAfi2KsiOuqXXjB5pR846h3MwbnF+J3+1kWO0i
xu2zjPZxrMyMWJrsEjU0Wc6chx6908QUfC5/nXYf0vmXi6uSufVOJt8CdnuP
BWpp+NN9U/hNS96j+qg4hg71KufvRKkZqmmrkMpzRTILWnhuEV/GiHy189fg
WXt4UVyGEfP8w+Nv1MofSfaBXPpCcTGeT+5FG2im+dD52zHbu/7J87/wgh22
8mhDh3m6xTsYkO8WP8B6+X3aDHtupv+YRz228SlOvmuM829Sa8R+9ULvdjHe
TIvI1puF+eZ5wO83rKTvFI/iNv/L4hVMyhtoO7DRTo7zO4GwtwJ+qcjl2er5
e3iKplh8AtX814qvJf+D/JzYj/s0W8z4Dl6xwzq197FcfZe4GcP0C9wvHd+5
49POWYcmZ/1nrmy9b5t5k7gRW/mU0eYj17c3xuN68j3pna5vmOaG/+Jfs+eo
f2/+e2p3MaG+R+0TVJqhXTyAPfIvxIM4JH9SXIM3zHZV3xEU6f0fvK2whQ==

         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtWmd81tUZ/ds3C7Lz8qG+eUdAdgKKVlYSSMIQEcFq1WorUFvrYCouIAgo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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1XuQjlUYAPCD/bQK7efSlF0+LGZy2aamMLUSKZTUSJNJLmMK1bCyQkgT
DXatGLoIm6Wm/qjcZmRoUsx00S7GnUVlQu6XZjWp0O+d88dv5n3e75znPOfy
nq/ViKIBY2uHEEaQonZuCB05nRXCKRbTQdyeRZ7f5TitWobQksrGIVSR1ySE
K5kQ/mKhuCcP01HchNvTIYxmFFeah1DD1AYhTKOKWi1C+MO7VZ6vc43H80Lo
x5GbQihjHouTNhyt7z1lrJe/kg7Ga091oxB6q/dRTqj1JJ1Nbpd+uxkudw5p
mmjTlK3ajKeY1uaVT3tzqpbvMPvk38B5ubuJC/lGzSs5fKuxc7RnkFwP0NB4
QxnC381COERLucdRRBvzyifPnJozhk3yf52soTE+5TNWiFdm4vOzDGKgMZ/h
YsMQllPBJWPuZVndEM5xlgJxn6QeNdzPGeOOlKs36+1DSq4sCrXpRlqbLVlx
LU6o9zgX6hiH7t4t1W8Jd+tzDzOYJJ7Ib/IdS8ffpoiXMdvzLDqzX3yAXtat
RnyF7+U/ySNy96LGOFnWpA6V6q+qG9fnPW1G0sXvs5nFaXGt3Hg+P6A3QXzD
+x6ee7KQaeL3KddnGf1oqo7byFNvlT3bzvPiydynrns57rfx6niVBtn2k6Hk
WPc7GWvNx9Ccxvpd9W6bPHek41n/US0/JN+NOXRgKjvl2kH/eiE8yXJe1OYF
ulj7rsyhr3gkZZ7npuL+fSz/J/zirL3DfLKN25aexuvBZeO3EedzwfNFunu/
UbycCvO6xEW6iwdn4jx/ZxFLjVnMl8a7xn/MMOZb1DdeA4awVr81mbj/M9nD
zb6XW/jV/s5nAc/JNZjh8uSk4vmaIH6N1Z6HpeJv9fTLZpI+k/mJUcn8Oer3
I6m4Flvs43d8aA+XcIbtajvFCnOt4CmWarOVP/1+mRUcFh+hrTOxT7yfMv2K
KTSnMuZSbsyPuG68ValYZ7m5ruOgeR5oFM/1XG1Kc+N+lXAoaSv/It6Wey97
qNFvF13NaQ6z6St+jE3W/GVe4Qu5Ps+NeapT8dyUuKsmkHHHFTGWEeqdyTC1
DqUhc+QalYnf8kCeZo061ib3jhpKKGVesrcUyN2pblzbKrkquSbPdfpbv3/0
+7dZvAvzk7uKPvJ3ysT7ewNf0Uiuq83iffYtmzko1+YW8S6cwuuU6FeaiXdX
NvU4Kz6TnF/r0Y6Jyb7Ld4ydatvBdAbI1Zly39l5ztFOvwcz8cwuTMfzn/Jd
tWWc+6KIDDu1Ocebck+ngG7iwkz8Lh5iAdXJHc4bau1EAUvE6wieb1iX1YzW
76VMrH9jOu5fgfiuTNzD3HS8U/aZw37K1f8E/flZm22Z+K3spY1aitVbylr3
yhqGU5HcmwzQZmVy/1NtjVex2x62Tv57+R+LVB/p
         "],
         VertexColors->None]}]}, 
     {Hue[0.5029270069767473], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV011o1mUYx/HLuT3qxqZbvtWmz+YySEsfVAYyi07C6Ry9WLqkF5wyZ8QS
lg5BnKYnlWOFGNlw4OaBUboFQbQVvbMtK5qI4hTNDkqd4ECXTT3o8z/4cl1c
1+++Xu7//S+re/P5xqyIqMdEDKYjfkH5AxF5pRG5+JPfWxzxFValIr6RP4/7
RRHF8il8T/PDQxHt6FSonj2ECn6Tc+34LCfirZKI7dg8OWKrGg3IVadcjXlY
OD3iAO0WtNKn1Zib1DLkv7RjOKjXZNop6Oevm6smbhVEXJsTkc3/Ij9iL/s2
Xp4aUetcJQ4URpxjz+JpZz9it+NxM1TqWYt8fb9z7iKWTLOr/v8hnR3xMW07
9tJn2GpsU/O0vkPYqO8/YgNIq3/EuQ584h5eM+8u/GHHE8l+uKTXenYdNvBX
O7cKfWq20jYjb4a7U+NT/KbOBDbwjDsplS9DsZoXzDuM3WZuUG8rPlCzkJ2G
PvOPq11Af9hsj9AW4EH31i/ejd/t1cf24g7/inMDaPHdO8WO4rj4iP43UKPm
PTVmiQ/pO9+3zUHpJPvoU4nNZjsm/zVOONsk9gq+FE/RTkI5fY/85zhNs1O+
AyV2X8PWYT/9o2wNKvjDtHeTb2CXHLGrybfgd5u3B0vM/L4Z27Dbvb2rTxM2
enuDZu7BXW/jnPxZlNjlINuNffQvyr+AUfdziv0VNXY8o/Zo8qbVP0k7jCH6
v9mfMca/7S2M4ZT30CK2B+vFc9gU1vp2j8nPxvFcs5n9Hbxk9x1sMzL8ifpk
o8Ns9cn/hxW+6XX5keTt2veYubpw2S5lYqkkZ86Z7AzM8ZYmOHdN3yp1lif3
iiJ32C82gHtmW8ouw0mzLabP4Cb9arYaT+n7k/v7EQ1TIt5TuwuvmrNKfhFG
6KeqvSD5HmZbLL8Ib5ihSu2V+CvZnTYLh+nXyL+Ob2melX8O2ebZxNYhwx9n
i+gb6T/k70renvgK555Am7Pl7MN4kv8/hNev+Q==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmmmUldWxhj97YOxuaPq04NfnnD6CUSEiNKgJxBGNMmhkUsOQLEFMZFRo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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1WdsVlUYAOADHQiV1YKMttwWlBGFAiImCGpMVDqIARfbMCzIJjIUFFr1
jwgVRBIoiGEYIVAUNTFQkDhYItDyQ4WILGWr4CptcTxf7o8nvW/ve85533PP
vV/u2OlDpjUMIRSTQuOcELJp2yqENuzPCGFDhxDKWNo8hNOcoqB9CBErDP6Z
n+iRGUJ3ypNDWM0VekQh3EZ2yxCmMoXXxc/RJD2E4Qwj2bg8fjPmV9YyIiuE
WYy/JYRxZPCUnH6kKXYEw8kXD8yM17tKf//rbNwd5DQKIZeZpOorhRl6msnn
fK2vg9TpqZ7CFuZTW14U17qLStbqdR91SSHUsplnxcX0dX1e/xfIMn9mTjz3
F6Tbwz5yCvnP/X/Zxpdqq6BdY3vBRMbLGchZ93ubsxejzfUMVeYpoZSheh3G
SH3eSlO6yEnjtDV7ysnjuB5OUOh/9fb4Jt2zQ7iLTU30QC3b9X+IQXov4qZ9
SFVHNzaqZQOP8614K4fVlW2fsyhUx0SWqWEpD3JOnKq3Tva8I8dSnQd1fMgx
NVRTyiVzXaTGfDcYmDgzchrpY5+a9zKbUeKRFOtpAp/STNyci+6vZBU1xn7H
w65X8DYV5t9GtflLeYXZep3DQ/r8vVkIf3BCvIc8/S9kAS8ZN5+bev+HQcau
Mv9Kuqu/BwvoYlxn2pmnPZNoqra/5S1XwznOMldcHsW9l9CTMcaVMlItI0jj
djn9edO5K+N+9qrjq8TzUMN7/MX34uOJ99C+HRVXJfbRuNoo3o98ChJnXHyJ
Wa6fpxPbxQej+LpjRvxc2ojbkmO9Kv1Xc0C8nyPuHyWSu0u8O4rP01YqWCde
FMXv8mY2UWAPOnGn5zWfebRwNmrUPEXNu6hMnCE5RbyWeFe4hzecn0WM8c6P
5SMGG/cyT+rzCZITf+3dvayz59e5xk51rI/iGm5Qk/jGiIeyWG9LOMlF8YUo
7iknI/4u9FRfQWZ8jh9Iib9Hb1lzOa9a7276cNI79CeHmtpf5rFMzjh6ul/C
Qrrqqxt99XTe/BdolNj3KH6uA/jFPt8npx8Z8loxnlxxf6a4nkwmV82flBl/
H99hENXq2EG6GsYyhnQ5Gex2fyrTmGTNyTSzH59Zdw8bxBuj+BmOYjQHrHGZ
osSzoV4fHeXkckWtV+lv7A05te3juTskx+ewoTUbZMbvcj6r+MHzOZk458b1
ojdn5ZxhoW9DCVXM1escduqzkrTWzrg1K6P4GT5q7x6hqzUb8Jh38l3WMlzO
nCh+p/I4pb4XzPUiWeb5xnyH2WLcUhbo6UhS/C0rEa9hk+v3eZodid8x6pyB
ArXlk6yHFJKcsY/t8SeUyfmA3u734oz8beIKfnR9ijz/n6i2gij+LUlLj3/7
lshZzBI5ZSlx/mp1lDNYDUNYzzU517lsnvLEt5QJ4uLM+LduQEo8x2rxGupd
b2ErhdYsiuJnPZ0ZtBbX6eOI59ChZfzb/D9hPyST
         "],
         VertexColors->None]}]}, 
     {Hue[0.6023416055813977], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV031o1XUUx/Fz895LTffgQ+rc1OYfqeSwQNCUIElBtmnkxCbZyofUXKj4
ME1YxcJ/wtYEqSVR8yH/SOdMwUo0H4I1RQXRBtkmpA5zZvaHMGUTfP3+eHN+
fL7nfM75nvu9JUvXzl/zVESswAB0PRfRhr5nIwaNisjFCgnTaTOwbFjEyLER
I1AyOKLZ+W4cVnyrKKIbHdmIXeIveJSJKHc+Ba18lok7MFn+Bh5v4k8++XLz
8HM6ooJWjtP0U+JfSA2NODcm4jzKCiI2Fke8hWFPR3zD7xNU8Ww241FsMv9E
WhYL9E2JEzDPd77zPNzhWZmcY5Da53mPx/t5EU16fo26IfZirn+TWrNNVTcK
ufbwmbgeKb1azPIbRj4TUUQrxmA5k8R5eMX3Eh7v4gW9riX3RoUZXnf+HrbL
eYk2Fxvd/SPaHkzg/zutHWPkfyBux1CzXSqMuIgHqYhG3gdQx3+Vmcsx0/77
5BbwaVZ70HkHLsg5ST+Bx3y63LsT9XbbTWtDkfweue3ol7+OxyK0JvcaHVGK
hzkRvc4L9Kqxn4nJvJgq53sev+II/3o5n6KaT5rWrVeTXvdo/2G+2gwtnWD/
1eJM/OR7Ob+PcY1nndw3kOFzm88/WGvOBtrn2EpvoX2FrfpWmGsuZtnDHOfL
8bc3UON8NfLlZP1uN+Vc8W7PJPfDfjtvpTXghtofaZdwll4hlmObeUp4jMMM
v9c7tGpcpn+hrjH5P6jN0LI4bc4MLYu97ttOu4eBw70VsRT35dS69ya8lu8N
ym1CL59a55uRY4Y29/gWP7jvI/0fYpfa2bRZ6HTHyerS+E6vq36rHqzMjZjm
/GUcl/Mhvy0YzbNRz/34Ut9i50VokDNWLMNO3z3e2l0U+F6s7m0cc99CMY0T
yRtL3jIqedba7Was9D/dZ8a9OGTn/Tz68Cqf63r2Y5v38Ic+V1Fo5ioe05Kd
8q9Stwid9lzmTi/ifznTxYXIsZ9z6s4jRX8CT92vhQ==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtmneY1FWWhquruzoDTSiwqruqS0mu4jiuMgRnn0cMBEfQdYAmSZIgKiqS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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1HuQjlUYAPCz9ttdVtauRXy7a3ebMqUtGjUlNVOh5LIuSSi7rFqXxLC2
JpduUjMhMmp0s0qTBtlV7hGlcqtmqqVIdypENZmQW7933j9+M9/53uc85znP
Oe9bXDG+/7gGIYRK0jjTJoSz1GeH8CQz6JkXQg/+SoTwNzcLTIg5VBDC6ibm
ZRlzdTKEjpRJWE4t1YUhLGBasxCmUsJVcvWiqzxd+EfOCvNG0D41hB/M+5Ex
YuZwUMwB5jPQ+C4G+92ITBbLP5NyuWtZQbpxMZ1zQriew/bRwbz21FhvUSLe
z05rHqW/3/1IYVaRfDRoEUIqd/Cm/S7hYNMQnucFCsw70zqEm9Q6lzl0tuYg
9lvzW2bST1xfVnm+mjR7nCSmir2e76Odmius2YsZzUPYTT01xouoVsMltGWM
eb3YYl62eU3ZI//X5Kl/l/yfMtpeR+XF53UL88gy/lfcaHHrWUeaXInonPTo
RQ4x1ZrlfKGGYmsWcat5+TxrD9/zHYPFDKJMXDm1vCZmRzLuZaBOLdvEbKdx
yxDek2sjtcYruNPvgVGP6RSdBY3NyWSDNXqLuYxr5X6apzjs7h1hjftXyUjy
xXSiwvPh5HBdNI8mfmc1j+tsZV4JJzNDOMES/rP/fVFfc0N4hZfpbN4QVpnz
LiPYrrZtLFPbcs6wzHh5Mu75VKZR7a4M5Lj3oqu70oUFej+Bt5zDSU6w0Zpv
FMZ39hxnecias7lbP4aQSW8xpWx15lVMYq/xpsJ4Tu/cuPZp5j3CpeZ8o969
ZBvn8KfnNSximHq7sV9/S9R7Oeli8orivmWzRVydemv5Wa2/0DHd2JrbaON5
AZ9Z/5hcaeIWO+/XE/H71MR/17DWGmuopE5PdtDDHbudU/pSblzGWr1aR0P/
DVVHFRn2kc6a6H2Qayk71ZqUvzU91dGBCfrxQU78ToTo7lFqvXMpIZxnu5jf
GK/ecblx7Qn1plIqT59E/F34VcxB8jzPZyyfqONjlqpjGacpNe7DLHXdwI2s
l2sD3dPj8z1FtnFO9L/cY7ifC+U/bb9f2v9F6i2OvhXRN4O+1gukMMt4NhX6
24xc2thXIfPtrSvd+F1MWlF8Xptz4z3eJqY7P3n+HPPYqY4D7FHfbmZQne8u
0bxhCC24jzq5VvKhfU6kinbGTTkkd6foveIP+Y9w1nmcS437+Jjxo2QYN2QQ
98j/ICPlrqQl08XUUOv5CsqiOyh/SVHcy6PWOcZCMa8y2PMhLOW8b+7nHE+J
e9CFLPNO6cNL5hzOjetM+q81G/3elBt/AwbI9ThDo7q4gIViHqbAPh9gLBnq
TecrParn4gx1yv82k53DFHZxRB0Z8g1XwzCyGSHXveyz3jPMZI6Yucn423AF
V/K+XJsL47NeKdc7zI5yJeP3cQqTyXJX2jLauzGKVkyUfwJ1cq8kRc1b1ftR
1OdG7hitSI/eS1LVlWAA++XaQkf3bzpP8D+gWBoX
         "],
         VertexColors->None]}]}, 
     {Hue[0.7017562041860483], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV01tsz3cYx/FH14MmVKtbq1WrhgunGBJzDEEsLorpxClzrKlsoTGCG6c4
RKMlJA67oCgjZHVIbBcoi82hZVlsS0TrQtLJ4lQLcyzz+l+883zzeb7P83y+
3+/vVzSvvGRxUkSU4gOM/DiiO77KiKjMj9iCFRIXCiPqkJQdsbprxCr0/Chi
QueIiRiXEnGVdgXtciKW2rsMPTtG1NJO4LD9n4ifY8yHEafEk7hEzzQnC+XM
/Eq7jNf0Btp1pCZHXBMP4BQ/f4g3kUP/25x7+Ia3vKKIIXjBwzH5P3HN/u3O
dBTfdYg4WxBxDjPSI6aZMx21/GSLffFMn4PqanBG7WtaE0bpeUSP73FPn23O
/TV2Ovvv8v8i356GvIh6PGwTkSL/WJ8v+Owj3xsjzHqrRyv+yoyYIj8V2Wad
kb+JG85eQhuCs+6kQZ/rqEiNOCT/I9Yk7t+5x+OXLGdPzE/cv/4F6j5Dldoq
+UqUeosHXSLu46f23oTHWuzjM19+AHroM1gsxKfWi8StyFU7UL/2KNNzjDga
t6zT+GqLI86YRnugZxZ9uTv+ErltI4bRh+Nn+hIev0Xwv0TdnkStO6xWtx/l
/CSbmYJ/3E94z2447W5vq7uIFrXF6vrjmbl1tOPYS18sTsUx9/BKv5cYau4g
2mDMpM9N3A16ufPz8nW4au5v4v94ar3O3hr003OLWIkS6ya+GjHJd1JGW4u7
embw+M56J5/P9c7gbSFvrfZ2THz/9k8SF2CH/fn25KGCtw1mrsdkc1d6n63o
1Y4uvxHz+RyorhPS1T6i1SNJ//m0UjTS19EmJ97J/s20cryhTxQnoMq6H5+5
eMxni5lPUMDDRT7vIJXPLHsz8cS62Tnqscm3d1x+F1b4HqrNWY/ZZu23txrL
3c8s3/VspPlHdqvbhf+871h756DRrFYzu1iPtE5S16znQrN+EGtRr38LPZnP
ssS/bF2BYv3fA7kMsbs=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmnl0VVWWxl8CIYyBQHLhJu8lL92FitZkVctU3eVUItiKA0NAURAkAk5V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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1VlsFlUUAOBrW0AUWrrY5S/qz5MUFxQjVEQ2IRoFlE0EUZayVEBKNAJ9
EFyQfdOE7QEEWjYBCZtoUApRsFBBY0BRkGgiS1BoCVsEBL/J+PClc/45c+49
d+6dNhta1mtsSgihhHqsSISwnMGpIQxhMwXifwpCaCtxJjPYlwzhN57LDeFZ
rt0VwnHxMepcX6CT3w94dn8irrOFlDTP3RvCCfpmG5P6HFT/EBdvC6GCShYW
hjCO1SZ2hcv8Jecsu93fQ6a5zL07hHmUNwqhiBZck1PD33LayXmC6+IbdHBd
6/c6zpvbuUQ8r1R6c/KeENLM70yTEE6ziA/krGS7HrYxjH7iKYn4uoRshuv/
eebk6JPjzJEzl3L3i2hBrbiB3tYYbzUvUl9crzCeQx8qyWsWQi5LreUyzrNc
D9v/X4MqyuijXm++1NtXNDZGR310YHh6CCP4jApzq2Sy99OSh8lXv4Craq9h
LefkVNPI9UBeZrD6QxLx/H81xjHeF79Hut8yGES6HjLYYf4jKaWHWg/Q3nrM
zYnXp53n+pLlmUz2qPeOnHeT8bx+l/MHWeJsLtknlaxim/6n0Of/vRLtmU56
fZUGGdYyI+53V7QHKFA7n2p5qWqlUKrOa3zBSvEkioxbzkS26KGGGfVDmE4x
Xcy3ayLufRazo/1nr8xjRJZ3RR5pahVE62f+d7LTGIvVWlIY7+VoT6+htZzH
aCjnDgbQWf1EIj5nv3CUEXJGJuP1+JCPmG3MMeQYbxglzFd/HgvUXsgp6vSf
rt7r6oyhabQ/xI3IdZ3HSNar/wnd9N6dpbwg7pmMx+vMUzQRXzfuCj1doI6z
zt9hShuH8Dk7aCXnUSZkhnAfzTlpbn8y1VpO42fK1Bsbja32eq5zXlwb9Wzf
5ZHPg+LMZDxeRzkd6K7+I7yl9l6+4XFxMa1dt2Fa9Jteixmlz9HsZHVTfxnQ
MIT+NCRTTlt2u19FGduMuZ2D1uNtJjFbvIEK1yvpxTFjVpFifr3oST/xmxR5
N805ai6HPLeZrz2Tra8s+ovHcUtPN/mUtfbyOs7Yy4tZQrGcl5Lx/Y28wvfi
H0hE31s1azishyPkO3vVztZ+NpjH+uh9mMdEqqmQU0N9efXS4u/bQ2q1TMbv
+mJ2fO66e2fduCTnMl3tq31qnWS0+6NIsEi8kQNq76ecrWotZ7x57WMvrdTq
wTPqPM3V6Duo1/ks0udpTjHB+xlP3u16oZRNau3hijqXWcUtcfAN22INhlJC
ob6aMsc7fJL23FD/BFN903/iCLs9W8UydT6mloS4TTJe34E58Tl+QzwzOg/R
uSAwWTws+r/n/v3iFuzS/yq26j01O94PN+X8Sxfz6soCfjS3dXxrzfP0n8ss
8SCa+20iE5jhuenJ+FzcyInP53fO9CaWRf/DOMd/TKwgkA==
         "],
         VertexColors->None]}]}, 
     {Hue[0.8011708027906989], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV1Hloz3Ecx/G3azJXtrHNLUlK7mtzJpJEbhZaKVeJpchRrmZI8QfJH9Qy
ZjNSGMo1R64xchS5ij9Gzgib+/H749n73fvzPl7vz+/z+3aau3TykroRkYt6
qGwTcRxZSRFn20YUI7lRxI52EduxuklEWseIXqhNjejSKWI4mqZHtGLT8aFV
RN8OEX2wskVEXbn1cEZ+TvuIVfjdLKJG7Amy5Ze0jjiEq0Q0NP8Tf0b9iPGZ
ERNwi7ZpYkNwmeA7bBWS5RSwm7FI7VB2GK7LmUXLOBTQtl9sE+bJKTfzJKpa
RvSjcSLW0Blir/nz6NzC34pxcsrNL0ReHbvSPgLl9B9wXoT1cs6wjxLwc83Z
hq5m9bVLP/yi84seX9GFtmbOP/Pb8W+yN1Ct/zH2NvbxJ6jrgxq1GfLTEzvI
n2zOQuxJi+jJ9sAo/ni7LkehfZfrsQLd9Bkt1hO9xPMT94o02ronfjc8d/+l
dirB2+YR79VdwE21M+XmIEP+OrG1GCv+wj09x3R3ddD80yiw+2P2PN7yL7IV
+MgvYTdjDP8eexdtzE1lU/BNnx36bceClIjFYtNRbK8kGruiQv6nRL/E/vzG
YrX8bfwydUdwXW0xjYewjs5MORmokZNtjxbIc4dP1T3DCPHBzqdipvspdX4f
1+x7Sb+XSKbth37fMUDtQ3Vf0FntIOdZKPB+ap1nqs2SM1VsIPLFv8ptqP8y
+Qv48/HKXofZMux1J9lqs3DaM5zDzsYffrXf/w2e+T9eFruCB/aqNKcIp+g8
KnYEe8QX6bcR1fq/V/cOr9UuEVuKn+Ip6lKxgs4857txsEHEhsRbRm96gt5/
/BM0t5Y7ElvkV5rzE+/MOue8NPEO5eeLncQU8QrfiVI88K3o73wAcsz9w/7F
Lj0nsROxU/w/uF6w2Q==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmnmYldWRxu/tbqCbZqf7Nl/3vd39CWpEkycKLtCKCzu4IiAqalQUZBNZ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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1VlsVUUYAOApFyhFQFlq6UXoRYqGmABiQCgKxiXsCC60xBgTI21RWRMI
lAfWVgQkEgmLYESplMXlQWkxAlZBkE0aCnGDQARFQEREpQU0fs3w8CV3zpn/
n3+WO6fz85OemNgohPAcTWibDOFB9nq4h+nMSYVQxE/tQuiVGcK91GvXkXd7
CP1ZTJp2YFzbEAqp5J/sEPZxNi2E++Xqw6EO2hxvGsKPLGZmpxBKGHZrCNdb
hXCDB9TRnylipvIlS3NCmED7NiEUU8TT2k+xoHUIpRwjQ/77uN44hGtsprVc
t5EjT4rJfNExhGpqW4RwlNyWIZTrc4jmYjL4NBHCFXNok4zrMY2u5BizL2U3
x+3NEX1qyRT3lbg9NO4cwiXrUmyNtlFFE8+aUu33ZKZwUL0H6Gc98ljEf+Ja
6fez92dYThvPbhh3Q8P68jcPG/MRvlXXKyzkT/2uMFXMnXRhnvZ8+tjDX+zn
WQ6KO0ClWqtIV3u2djIZ8/SlH93FtU3F8R4T9yjHta+xTO7XeaihTrU18myb
flUUMUq7RyrGLOcNWppXC5aIeY0TvG+dZzPEeVnNKqoacnFMzaWUUag9OhXz
nG4Xz+Y8cc/yr2M4QtxwsuTvyYAs55qLxuinnU2d34M9G8QWuXZyUe7fWMsO
7e2s8ns15+ghrjsnxZ6ip9gJ+rzMBjVUcJX96jjABTWsYS254rrSUswOsTsZ
qU8RHbxPsl/tO7V3MEN7JvvYJP9GBqlhcGasqUKfz6nx/jCzqdSnnDne13CY
V7U3p+Kc1jCMFc7ZSv7ypy/nPXbJdZlce92FWvm+1v4gO+7Dr2nxfzzMHIbS
2zz6UMYf+lzioPeH6Ch+vfPzLuOcqcJEPF+X1VFDB3N/iRc56bycYqxzkkFz
5su1gCflStCYI+K28o36O4nrSJ0+9dnxfJ7T5zx5autPgbqakcH2O0L4jOYZ
IdzCM9TKdTQV/xe1ch5lulzvUHJz7bvRXp/ejLWvBTSji/yBj8Um5G9Ervtr
IFvdXZ/wAoVy5VGle7pcTRlgTQpobz2yEvGu7WX+Sbq5Q2Yyg3x9xrDL+920
07eZPUsn33+0gI2MEjea/WJKmMUk9S5Mxb0eyjU1L3XPzeAe99wsSqiQfxHT
EvG+2s0Kubaw1926h+k8bsyR1BmvniHOywXtGpa7p85wmvXGnJuK34i5mfEc
FmuPT8W7ZiWrKDXmgmScU2YinpHF2ksa7li/7+Ju1lm7t8mxbikmky/XxFSc
Uz2bGG4vRrDOPoxiNBvtxTLe9E05zzmy5L+aHb9DZZQy1rMyxhuvOBH35UN9
PuKttHiXjOSENakm3fnMZwzV6viBgc7BAH433++1v0vF++J8ZvxG9mxYEwar
tU5cPf8DChMViQ==
         "],
         VertexColors->None]}]}, 
     {Hue[0.9005854013953495], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{8522, 8523, 6021}, {8242, 8275, 5197}, {6561, 6518, 
         298}, {8438, 8407, 5684}, {6647, 6679, 565}, {8241, 8240, 5193}, {
         8454, 8453, 5825}, {8440, 8441, 5770}, {6790, 6745, 793}, {6525, 
         6524, 324}, {8404, 8403, 5668}, {8507, 8506, 5977}, {6402, 6401, 
         91}, {6522, 6523, 312}, {8435, 8434, 5747}, {7120, 7147, 1683}, {
         6641, 6595, 468}, {6601, 6646, 486}, {8047, 8081, 4638}, {8525, 8524,
          6049}, {8358, 8377, 5516}, {6458, 6502, 182}, {8494, 8495, 5935}, {
         6573, 6600, 407}, {6493, 6530, 252}, {8513, 8535, 5991}, {7802, 7801,
          3916}, {6963, 6921, 1195}, {8323, 8322, 5431}, {6861, 6820, 954}, {
         8017, 8016, 4555}, {6453, 6452, 167}, {6794, 6823, 884}, {8300, 8301,
          5366}, {8437, 8456, 5754}, {8020, 8021, 4564}, {8376, 8375, 5590}, {
         8360, 8361, 5527}, {8380, 8381, 5608}, {6450, 6451, 139}, {8534, 
         8570, 6069}, {8485, 8484, 5902}, {6674, 6643, 550}, {6569, 6570, 
         394}, {8460, 8461, 5852}, {7143, 7117, 1676}, {8192, 8193, 5045}, {
         6503, 6478, 203}, {8567, 8583, 6145}, {6744, 6718, 712}, {6676, 6677,
          636}, {8406, 8436, 5675}, {6527, 6572, 328}, {8571, 8538, 6090}, {
         8584, 8575, 6174}, {8408, 8409, 5689}, {6749, 6748, 802}, {6746, 
         6747, 797}, {6487, 6488, 228}, {6717, 6675, 631}, {6596, 6597, 
         475}, {6599, 6598, 483}, {7616, 7617, 3363}, {8490, 8458, 5845}, {
         8457, 8488, 5833}, {8577, 8628, 6218}, {8514, 8491, 5926}, {8536, 
         8516, 6008}, {6490, 6489, 245}, {8325, 8299, 5363}, {8489, 8512, 
         5912}, {6897, 6898, 1118}, {6591, 6567, 384}, {8297, 8296, 5352}, {
         8083, 8048, 4642}, {8160, 8159, 4954}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmneYltXRxp8FRGRZyu6y7/vsvu/uPoj0xUgRC5CEosaIUdDkixSRookU
DWChqgvSIsYgmCZ+XyiCNBPQ/BFLIsQ0BY0oxRbFEguQBGtU8Lt/mdsr+eO+
dvacOXPmzJkzZ848bzbumuFTmyRJMqRpkjTT37tPSJJVwmHhkNBEaCqMEIYL
U2v0tybo1ScEfXpN8NH2oOgVwlHRg5pHG2PXmP9WzbFQ2Cs8J1ygtmGek7l7
FJKkldDrxCRpFAaL7ibMFb1LeEkyXxZ6im4Q3tFcTwnL1XZQ2CV6u9Bf9G3N
gwfeeSeGjMnqu6gm5mU+ZLcsBB/9h7z2Ac1DBrpdKNzzX3Y5IgxsHjys8z1h
cPNY71zPtdv6IvutmtCbOVgbOqMra0TX5TUxL/N9o5gkZwsDSpNkWWnYEptj
O+YqOylJWgtjhNHCI5K3vhDt/H+f6B8KO1okSbn+/00h2qDHCc8WYgy8Wxgv
vkdbhExkTBE91W307S2EDMZuOylkzxamtYg5mPsHheBl/GdyomPNwm7YeJvo
7c3Ch/CRWXK22cITwp8N6Br5X7XwJ7dBp8J80QuEZ4U95n1SKDSNMWM191Dh
Gdmyg3T4SnWSVFXHuD2ea47HMHZmk9CBOZjrRrdB/1HoLfTxnMhA1gdptPP/
avvye96PmZp7jFBsEbbDNuiELtgo0V6+q/+btNRZEI7JRseFC0UPE46r7+1C
tPN/eTHaoFe1DB/Cl2pbxByslTPC2WDNq+ybR+yb+BI+frfPL3txvFn4MHuC
Luwpe9nEOqDLPZ4Pn9hqHdGpshhrQPfVLUNXfGiL+0fbF7fan/AFbDLF/oNt
OMOso+g11NrH4OlUjDUz9+GW4fu0QXMGVlm3I+4foL4uwlHRg0qjjb6BpcFP
HzqjKzzLdKZud8zgDPbRXp4ubNCy1idxhgeaB94vlSTJacItws3CR+L5WDhP
9LnCh0m0QZ9j3l5Co8d8kAQPfUOF3kIfYYEwX7hBc9+IDqL7Cq+J9/UkeOA9
mERbb8tdIiwVnhcOCDep72bhaeEpYa4wT9gt7BJykp0Xvif6WqFGdEG4TvQM
oV50JswSPVOoS6MN+kYhFV0tTBc9LQldZwr9SkLnctEVwmT1TUqCrhSmJNFW
lYYOzH2N0D6NNuipSfDSBs0YdGVO5kJn5polnFESc/ayXebbxn1st1ttU2xx
i/AX2+SvwitJ7OOpwqtJ2LSX92qx6CXCAWG/sElzzRWGqG8Ftlbb94UXhOeF
i9R3sfCy6Jc8dqn7kDFcfSPSmJO54aUNmjHfSEMGY18UatOwOba+Qdgg+r40
9niJbcy6FwkLvUbW+pzwrH0O3/uRcFdJzMsaWduXzMu4vR6DLGy5+L9k8nef
eRZ7Xnxrf0nYAb9daXv0tU4LLXuodbjLPIssY79lFtNYI2u7Pgnfow0aH8S3
8El8ER/jLPbDBkmcya6iuwnzRTcKXdJoa/Q+nyy6Yxp+P0fonAbPLT4XVwjj
hG3CL4UJwkThQeEBj5nrs/KkMNtt0E+Y90rhVx4z3jIYu92yx5tmjk5p6MDc
nE107y4sSGINFxcjZhGrlpdGjkCucGfzyFnIXciZBvn/HhrbkEaORK50Shpz
3ORz3lP0qWn48SL8XPRpafgkvsncyGAsOiCLMfAiE95eafg4Y5CFDMYis7/z
mmXOn8baptjyF8LlSbRB3y8019gT02ijr63odsLVor+bRF+LNGTAUyK6qTBK
9EjhJNEt07AptqWvmTA6CZ4T0pgD2WOSiF0d0tg3YhhzEZOIRczZJg0dmPs7
SchCBmORia+dIWxMwudapzEG3quEVqLL0vCBiR7D2K3CFus02vRmr2GU6U3m
vdy2YQyymOMq+xW+y57OS/4Tu9lbYjkxvDQNHSba77ANbRPsd9gSm43zvmAL
zsQc+3JX5SjdnOOQ15Dr9qmJvIx8bK/PaWf1dRJGZVqPsD+n9lzQY4QDuWhb
p/znKaFM4x9RvrVB9F+ECv2/U/+PzEIGY/cKG4VNwqfCJ8JmYYtwTPhMmMbd
lEbuzxsA+lrh2STa/uVx5+eT5GvC2eo70/Zjjxao71bhOeFZYX4u2qD3CDcJ
NwtPC08J35Fu3xVeF/2acFUWbdAHhfGiJwh/Ff2ycGUWPPS9mou+icIrueCB
hoc+2sbIFkuFGbJFx6Yx9y3CX6zD9DTWzFr3JWGbzbYFNppnndF1t9DoNbGW
Z4RxWeiIbi8JM9KQiSzuL2x3jbAnCRt+KJ6PhHNlu3OEo6LfEwaLHiT8Mxdt
0F8VPsjFGHiHCv/IBQ99X8nH2PeFIfmQ8R32Io1YScwcInqw73vu74n4ahrx
kvh5ZRptDzqmIos5mQuZh0UfEQaKHpAP+u/Cl/PRdjNnRfiXxn7CGedspxG/
OS9TyT3SuP+fScIWtEFjE3jRGV0ZMykNGbt95hg7OY3cARnQ8NBHG7pgE2yB
TtelsQfYnvsfW38snJcPm0Pjw/gube+KPiT0x5eFK7LYU/byRWGkfGexMEW+
U2waZ+9y4flcnMHVwhrv4VHTa70ntP3cPEe9b5M1dorwlui/4X+iZ2SxjiP2
zUb7Fj56p2LDAzWRA/OGuFj0lJp4U/Bm6Cd6RE284XnD9a0JHvp4V1wt2ZOE
NyXrDc53FnMy12Hrus57js73Cuvto/hdXmv/anW873hzXaOx12Zht3dysZap
wtu5WBM0PPTRxtquy2KPWOP3stDhsO2+QbjPe/Kx6Y2OMbSxtmtq4k3HGtdZ
xw+t83rL+Ng6s1ZsjG1ZM7bChkf8DmtVEzbDVrx7W9aEzbDVVmG36Ldr4j3C
m+NXop+uiXyAvGAcMUnYLmwzPUF4wG3QE4UH3TZbmCM8KTwhzHVMIZbsyoUv
jRVeyIVPXZGLOZD9y1z04ZP4IjzESvaUvSRm4pvcAcR+fHRoGmees07OPtYy
kfULYZQwOhcxnzj3fnXYBFuMbxZ3B23Q3CGzvAZ0/zN+XB02w1ZjmsXerKyJ
9zZ7BI3NsBVtc7zmXbYBc48RtlqHkdYJXTa573LhfvPg++zJ7X4H0jfWa4Fn
n3PbLk3izjwnDRuwdvLX80Sf63yW/PZq2WiS8JDwa2Gy7Yb9HjaguU9bCTua
xj2KLcqbxTh46CsV1soei6pDLvIKjhNTLPtk3zvcP9OFDm6DniYs0djR1dE+
zTWTgsdOctt064Au91bHmGn+f5J5H/aauHe55zsplp2SjxhCLOkgOhPWyBar
yYFlsx4lwUtu0DkfY8gpiGtdRXcRbkkjxhPbP02CFx76GLOivfZceEN4XXi6
OnREN+x1mfd4k891YxoykfVZErr8nDdFSeiErtytJ+dD539I1u7qsPVIYZz9
FH+ldsWed3Gd6Abhdcl4g3eR60SvlURbH9eO4O3q3IsxvNtfYG7nZdSS/pDG
uN5u6+46FTnbSr+t3vQ8K/zOesPz7PdbravngmaObp53kfP4/c439rh21tF+
UW0fYF/xKWq3pTWxVupzWxynjruOt7lZtEGTQ8L7XnXYiTGjhNHm2WyfbeX2
kcYo920STqoJm4/0/6N9zrdaRrnjwnjvBbLKzIcc5sYHyt1P3odP4osd87G3
tEGzxwcc97rlw+fG8PZII38nPx/L2yKNfJ03DbnL+DTed+QwF4j+ehrv9ZlJ
8CJjq3P8YWnwzPR7hHuJu4u7/izhMfX9No3aEPUL6N+kUVuibYvj0PFc5Mjc
Wdxd5ELkDOgyLo23JjpdkYbO6MqbjD7aoOH5quivpFFbocYyKI02aGow5Gq0
QZOzDRDd3+8l3k/QZ/s9RNvANHjo442F7C+nUethDmxzfhrvfWy0I401s1Zq
NvR9LY33PzyMHej3GjLoI4YSO+G5V/Q6oWdJ1G+g1woNJdF2VPQ/06iFUN8g
TyeX7y1b9cpHXk4b9Gn5uNfItRpE98gHL/dan3yM4R4nrzhDdL980OQVZ+aj
jTseHvpOz4cvcU92z4dPIZvcvGc+5qCPOZkLHubiLu2bjznfcN6ALNq4O7hT
uEu4Q5DFO+PUfMjkjcGaWMuX8nEvcdeWiE6E9joLX66O+jZ1bnTHB/E91nC/
7zJ4P7ev0fa5fe7buYihxE5yo8ely/tpxCTiFm8YYjyxvT4fuqETuqAj7xZ4
6KvLx15xBxD72TPyEHjoqxUuUd+ladTkXk0i9+MNRXwiTlHHaXQuTz7+LeUi
l/Lmkoyr0Ff0/wg3ip4ptK/XXgvDKnV+KpLk0YL8ohD1aOrSJ9VKd2FMa73D
hUOKP2X+vsQ3mH2iD7k+T028VLwnC+PFW9E6crXSQnxDImebWNQ5F3q20v+l
SZLT3H2ESzX/CcKvRe8VulfJfrqvasp13wqThUnC+rokuaMu6IeEx+qiDbpU
+k/X3xnC74Sdwq52sqFQK7poWfydYpnXauwI4RnxdCgP2VPrYl7m2KP2Z4WO
ok8uDz7aoOGvFW8/Yb7aeguXSPerhUNay92VYct3CvE9AJv+thA22+bvSdhq
oLC9ddhsYyF46OObE7bvL2xtHXswRbIXCefINh9KfkFz9xXmae6e7cKW3xTW
VYZN18vWjwq1ZbJDq9irEn+PYM/Q7fNCfMNAx8HyjUFCJ3IOYUMhfGKMv5s1
iD7H34v4hjITX8oityafXi5dNgiP2MZz1Tcni/yU/PJ4XfgcvrZFWF6INbNW
vofdp/4VdbGP7B+2GSU0bRM2uqMQOqEL39hmZaEDc5PTo8v1WeTn6NRdvEMK
8Q2I74rY8jLheOuw6WTrib4Pe88nm37Ie97RPoVOG6XzJuFT4RPhoHTdKVSI
Hi+0EsqE0cIoYbPXyXo/qwhe1sjaGNNbtqgSmmqvRghfE84Xfir8RBiuvtPq
o53/m/mcsLeXVIbPt/JcIytCZqUwwfpssg6fWedjmvsV4bKKWMuLbRWDhQb5
Tg/hAfYyi/c3NaQWstVh+csj8p8y2evS+tAZXddY168LP7POl1mPTZb/eV3Y
CF22u+8Lu8BTWR86bbQ9x1v3B8w/yrbc4jED8d2ia8GlMfcFPmvo8JltPczt
B7W219rG2ewlFDVXSV208//pddEG3cj54ezXhS3m2CbYZq7/b/A5m+e2h+wr
7AF78WS7iDlfxBpiycXCE+0ipvAXHvr4v5fjxnzP32iaGEOsITahY6P/327b
4Af4zKfe2wu8D9SoFzl3pWYG74MV4TNfjGlm32EPP7FffN17+bDPQpn9aq7X
utvrmuN17/I68J2X24ZNsM1Ox95K++I0+c10YaewQ+ivvbtIuL00vjku0P7f
K9ymeNZf+KV87xdZ1NOoB63EN+pCJmdwViFiwKP+Bnq/eLdmUQujJsZY2oa6
frctC5nnup6Eb9N2nuujxJa5wog2EWO2qG9zFvU96nXEuiv8PZqY92pdxLAt
9ktiy+WF+CZLjPmhxt6RxVuN99dS0UuyuL+5438g+vYs8mty7u9nwVPv9x99
y7LIveFBFmNO8fvvUs39vbqISdxLd6pveRbvPfLzlaJXZJFbkT9Bw9PV+Ra8
6NjZ70dsx5oHu57I2jdlUb/EBnfKNpuEI7LNPcJ89TVmUX8l316QRdsx50M/
Ef3jLHIv8ptFohdmkT+Rc80pRMzf4e/7xOLZwoVtIibfmoXM486xmOvmLGq9
zIkseD53ToZtF2eRG2Hj9aLvzSL35w0AvS6L9wRtG7LgGeD66BrRq7PIVcn3
4F2bRf7HGGh4zvR75HfS9VXhdvn8wLaxV7dl8RZmz/CVOf59Bz7ze/G+Jtwp
3sHCRvHel0UtlJooNDqhC23YHh762AN8C5+f6u/+zIXPZH5/l9RHDsX55pwP
l29cUxf5DmeWu769v+dz53PXVxTjNwHc+Z2LcSY5i/wO4N9nrS7OL2eO3GGf
MLUscgju9h11ER+IE03qIgcihhFnk7rISYgHxIUjxcghyR0f1fiuxYjhxG5+
X0Asv0o46O97xAZyRHJDYgS2Yw5kY8N/FCPHJLckpuA7+Ci+iQ/dqf77hGeU
6zS2ilzzh8K8VpFzHiiGjJ2OS/g2e8pe4uPkUthw9xcxXn0rhTVtIkbMKcYc
yO4lbKmNPWZv32sTvoENWDs+wtzkYORe6IAtsQm2wKbsTVf/HoM9uiuLM9zd
761Vou/O4j3Ge+tnon+axfuBN83PRf9fFu8h3lj4Mm39/P6i73+zeCvB86Ms
5ujh9xZ992TxtoIHmjl7+73FWWZMg99T6IIOp/m9iG2wAWvHRuhGDDjV7y18
DR874t+hkBsT84n15MiPFWNP2IuTy8L3XquLHAAfJNZxB8xwHsZbgpxunn+X
xlukwb/n4U0yL4uck1yTmia+xZ6xV/gYuejsLGql5KT4MjF9tPO2v4t3fzHu
Le4rcjNyGO5d7k5yob3ClLLIiS5xLka+vVaYoXfLdcLjwu+Es32vcb99X6jQ
nJW5qENTmz6rffDQt1Ror7aqXLzXqFN3odbntxpvtvJcyGDsOMtqb94J5u2W
i7rwTMvK+e2HzO68rXNR96VufGb70IG5lwj9hDOExcIit6Mbb7HnhQ8q493B
+2OoMKk+ckJywfcqo42+u1x3RNaZlv2FTP7uF/a5fallHxDWS9ZCYUj7qF2u
Ez2/PvRi7BDPu9L9iy3jgGW2s42wDXX1gcTaLOra1LFbC21yUaemln0euW4W
30D5bnhuFm3QfOc5J4s2aL79DMhCJrKom07xOX7U/nAWd08WNQFqA4vlA0uE
A8J+oRN3r9DWenTMoq2NdSsV3Up4kfevfel64ff2qZ7UNnLxzZPvFOh+fhbf
dFlDQy546JtrX+juvcYnmLedbTPWOrQ1fbl9o8Fj8RHWwppHus5xoehvZPEd
je9m0Bdl8Z2Ktguy4KGPb2s/c27OWX+X94z6RmTxXYjvRIy9OIvvRsiAhoc+
2rDdUuF52/CnzvOR9Y7fIJy79+1/306laxq/IeO3ZPwmht8p8P3zY+Gb6vtW
Gr8Z43di681D30fm5bcNn3jMAOept/v3pOh+SRbftFjDPvnmQ/VRW+9TFX7M
WaF+0a0qfBLf7Cq6izChGL9RRdZLpTGO2nzfqhjPbyA2u/ZODZ7fPPHbJ2pB
1IT4TQq/TeFbP9+r+Y3MAtN7XPuZb5oaEN8C+aZHXZS6KrL4XRW1QGTy2ytq
Sq8m8Vso+vjtFbVDeF6wDsxN/Z/fbdEGzTcBbLXJumIzalPYGNtSo2Jtbwqn
V8Uaof9GnKmKtgM++9gKG2FLYsAKf69Y6Vjypu3UxXwzhRuF6VURc4m1O4XO
tjN9NwhvySferowYRCx6x75ztuMefvOB4xaxZa1j+Qf2qXftu/0dp9+2353l
uNbJc97gc3pKVbRd77Pb0W3XWU/W8iPbgDXtdRzs7HH7HMe6WO4e2eK5+pCB
LMa9xVqqwoZ9bceFwq1uo4+4s8i2wrazbLOf+J3+ju1yuscssgzGnumYhYwT
5DsnCjtEP2a+xY5l+4ShWcRIYiO/3djrdmpH1JBG+K251mf1ceUKm4WjyhUG
KT863CbyLvKvAcKhNtEG3b9tnBPO3ss+L4wj1yDfYvww533kf6uE5ZI9S7hA
9N3CK6J31oas24Q/SLc/Cnnfi8RV2nK+K+/V2bxDmKt8YZewzHncy66TbKD2
4PyPHHOFZM8Rhmuu1W3CD4nT7X03U0v4vDbqLNQSHncsr/K9vcV1KN4+x4Rx
rgNS69omPFIMndClWBbrwDbLvJ6NtbFm1nrIOqyxjbDVt9V3dm3Mg/yDtbEH
2G65bTjYuTX/oys2QzY6r7Jtj3hflpuXfPygdVhm28B/zGthT4Z5HwfahvBt
89p4azdpE/s0zHvHGvAxfK3c99Td7jvs9Q13Hr7Ga23id/sI97HX2ARebIWv
sUerbY+RogfUhh6MZe9oa/LFeMeSSudZ/64V1ca6We9viuED7H2HstgXcnve
RuxPy9rIIckdy1tHLskYeMkp9/se6+rc7nnf8919V+N7M4WGVuGD5Jr4ALLJ
NdjLemF069hT1tKhNvyGNe1wrYX3UblRYZ+Cp6XO6kf1ESues61Zc4XzJmSx
hnLz3+q4sNf8Zf4OQC0aHdCFGmGZ/7/RceZPPmezpfuVft+Rg7M27kDOMmvk
LM12fs6ZOuC7vptz2aLfnVO9/rl+R+32+bzesfcPPlcLHaP2Vf0nFmFzbE1M
esmxhLdgg9HTMpHdg7dOFjkUedANvkv+6DnYK3KzHs6r/s2bRU7GmA5+x0z3
fj/i3BCbYSNscbHrl9hktG251fv5/6K7Yyc=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1FtMlmUcAPAHKqsR5gGQj5JAEgxsuPkxsVbaUkuTakZOQbxopdU6XKAi
hwmlSB66SJ2urZsuDBS8qGUHqy2lNchDWwIl2Nqy0lYXKSc12/q9ey9+2/N8
7//5P//n9OU/9/qK11JDCMu5hdU5Iayix4+9JG4K4el7QkjSNDmEU5xkgZgq
pvt+N6fE/iLmOGlTQ6imikt3hXCR3RNCeJuf6c8L4TMGMkMozAphJkkxc+m8
OYQublCSH0Ix/b4PUDgthDnGZXBd7mUZISxljToWclYNBWqZwQV1BHFHxX3M
81FNiRDKaPdpnDGa9KsY1V6cEsIirhjXx73m3EAtG9VRwftquMw/pOvP5n7t
2ZwXt0m/jpTsED7020d8r7bLrLWmGtJ5U/6XuKT+cvswj3fUvIdXp9hXculR
Wy9/qGs/B2jU72CH9luUc2h6CF9TkG79/HBHCPPFPMAn1raO9QzrX6HfmAEK
7dlt6r2dXvXX08AzaqvkPfUtp4JnxazkqDW9wDoetq6HqJNnc2p8b9qM6+CA
Ne1nCe3m/IBSc86hhe1iWhmWe4RF4orkn8BxNUyWfxJPGldBg/obOclc/ac4
rz3ELtbo13BDu4sjdOufYKv5ttHHt/qd7NP+jQuUmnMxLeZrppRDavsyOiN1
XcyM19Nl3JFEfA6P8Tir9Ouiu6WdjHKad6Fcj7Benmlkc1jMNwyJGWRndN/E
PEqR77PYHMXbz2y22cskZYwZN56I27+nxPdhgXEzmWjMWmr4yxv6m1+9sz3s
ZZaY+/LjGnqy4nP+yrp+Yr52eVa8xjPmPM0x7+cL0tzTfe7jXma4iwXU8qlx
B2kz5sfM+B03y7+F0+Y4Q6773ybX9pz4PySHl7lVzHBe/Ka6OcGL+it41x34
MyN+E+3GddBtTC0bqM/1/8R/E92HO91LKuV7kNXRedHBObmuslPuHVnxGqui
GOp9b+A7Ruxnhjma7GkjJWzxW3MiPterzvEa86y/kj7/e2fZylTjpuTE76co
Nb7/18yZZo4h8w1mxfNvkmsj5+QZpFjeEXGjefE7K6aElfqvMGbtoxzkCbmq
GVBrP618LuYYQ/Z8F7sZj8aRlKeM1uhMzZlKtTk7OUyLXG+wzNlcl+tf/get
auBC
         "],
         VertexColors->None]}]}, 
     {Hue[1.], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{6571, 6598, 403}, {8325, 8326, 5443}, {8489, 8488, 
         5913}, {8513, 8512, 5992}, {8537, 8538, 6089}, {8457, 8456, 5834}, {
         6720, 6748, 722}, {8276, 8275, 5277}, {8437, 8436, 5755}, {8438, 
         8439, 5764}, {8378, 8377, 5596}, {6746, 6719, 717}, {6596, 6570, 
         395}, {6519, 6488, 229}, {8534, 8533, 6070}, {8515, 8516, 6007}, {
         6897, 6863, 1038}, {6453, 6491, 166}, {6676, 6644, 556}, {6568, 6523,
          313}, {8567, 8566, 6146}, {8490, 8491, 5925}, {8082, 8081, 4718}, {
         6490, 6524, 244}, {6402, 6456, 90}, {6479, 6451, 140}, {8574, 8575, 
         6173}, {8083, 8084, 4722}, {8577, 8576, 6219}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJxFmnl8ltWVx5+whJ0Esj0vWV+X6lRWrXZRDMs402krISyi0DbotFMFJaCW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         "],
         VertexColors->None], 
        PolygonBox[{{6598, 6599, 482, 402, 403}, {6677, 6676, 556, 557, 
         637}, {6597, 6596, 395, 396, 476}, {8516, 8536, 6088, 6087, 6007}, {
         8491, 8514, 6006, 6005, 5925}, {6570, 6569, 314, 315, 395}, {8628, 
         8577, 6219, 6299, 6298}, {8377, 8358, 5517, 5597, 5596}, {6401, 6402,
          90, 10, 11}, {6488, 6487, 148, 149, 229}, {8436, 8406, 5676, 5756, 
         5755}, {8407, 8438, 5764, 5763, 5683}, {8535, 8513, 5992, 6072, 
         6071}, {8570, 8534, 6070, 6150, 6149}, {6523, 6522, 232, 233, 313}, {
         8458, 8490, 5925, 5924, 5844}, {8048, 8083, 4722, 4721, 4641}, {8512,
          8489, 5913, 5993, 5992}, {8081, 8047, 4639, 4719, 4718}, {8456, 
         8437, 5755, 5835, 5834}, {6489, 6490, 244, 164, 165}, {6451, 6450, 
         59, 60, 140}, {8583, 8567, 6146, 6226, 6225}, {6747, 6746, 717, 718, 
         798}, {6898, 6897, 1038, 1039, 1119}, {8538, 8571, 6170, 6169, 
         6089}, {8575, 8584, 6254, 6253, 6173}, {8275, 8242, 5198, 5278, 
         5277}, {8299, 8325, 5443, 5442, 5362}, {6748, 6749, 801, 721, 722}, {
         6452, 6453, 166, 86, 87}, {6524, 6525, 323, 243, 244}, {8488, 8457, 
         5834, 5914, 5913}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1EdsD2AYB+Cvdltbi9LqsEptigNR1JkutCVEwoEYJ6P/2lp7dNh7tf+2
tjogITFvuBhXbkiIUW54GofnO315D+/7e9/0JavzV8WEELI9MUkhtGEyk5jI
BLIppZgSytlIE894Tgu9+4WQwwIWsps6XvGD+P5qks8qarnDSz7ROTmEEcxi
MRGqaOAhb/hMSAkhgUymkscyIlQT5SHv+ErcgBAymEop5dRyhad8ICZVWSZS
wBpqqKWBx3zgL73TQhhPESuooJJ67vOWX/wmLj2E4WQxg5nkMp9iSohQzkY2
sZld7GYPe9nHfg5wkENUUU0NtRzmCEc5xnFOcJJTnOYMZznHeS5wkUtcpo56
ojTQSBNXuMo1rnODm9ziNs1sZQvb2M4OKqhkJ+tYy3o2UMZcipjHdHIYRiax
dKJFH3/yhnvcpY4dLKeQAsaRyB8zes8jolRTxWryySaZ0PrP7J/QRA0RSphC
OrF8kaG3PKCeKspYyhymMJRe/JXPT7zmAVEOUcYicskino8y/4JmalhJHtnE
8d3OfOMll9lFKSVMI5EEftq9pzyhkQaiRChjA+tZx3zmMZciCikgnzzmMJvx
jGMsYxjNKEYygiyGM4xMhjKEwQxiIBmkk0YqA0ghmf70I4m+9Gm9HySSQC96
0oPudKMrXehMPHHE0omOdKA97Wib9P+e/QM8H6pS
         "]],
        RowBox[{"-", "0.004`"}]],
       Annotation[#, -0.004, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1NtPD2AYB/CXzZkbh5FDMqcxOUxbU/o5pJSkVIzpoJSKzmHLnG3a3LCs
PwARTVc2XGgKMy4y5nylrQtMNmIOFw6fzcXnuX6f5/k+76yS2pyaISGEBcqK
qBAS2E0ZpeyihGJ2UkQh+RRwguMc4whHuUYnD3nEd34weWoIU1hDIadp5wpP
+cbYaSHEk0strdzgCR8ZYNz0EGJJpZhDtNBBN68YYMiMECaxgAg5lHOIc1yl
mx7e8Jkx0SHMJkI+h2mlkwf0M3RmCNHEk8cW6mmlg/v0E2K8gcnEsZUqqmnm
Cle5TRevecNPfjFmlnkwjkXEspglpJDKetJIZwMZ7CCfAgopYifFlLCLUsrY
TTkVVLKHvVRRTQ211FFPA43sYz+ZbGQTWWSzmRxyySORBFaSRIRVzGUO85jP
cIYxgq/6H+Q5z3jBLW5yiZOcoJIKcljGUiby2/z7eEsP7Zylhs1kE8c0/tpp
H/fooIWDNLGdRGIYyScZeUkXlzlLE6Vkkcg8xvNHBj/wnC7aOUMThSSzkNG8
l+t39HKdFqrIJo7ljOKzG3lML200c4rtbCNCEhMYz1d3N8gX7nGXHrq5QzuX
uUQbF7nAeQ6wn3000kA9ddRSQzVV7GUPlVRQziYy2UgGG0gnjfWkksI6klnL
GlazighJrCQx6v//9A+nkKnv
         "]],
        RowBox[{"-", "0.008`"}]],
       Annotation[#, -0.008, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1ElIFkAYBuBRgwrKJcv8zVJbDnWw0ha1wKhDt/btIkFBBZFGJy9G0WqW
Zou2Fy1SGS2WrWYmRJBCqQRh20WiQ6YdgrL1iQ7PDAxzmPnm/SZjVdGiwqgQ
QsSwLRLCdm5xkzpucJ2rXKOVZ7TQxze+E0kJIYXZzGElZezlEu108JXBI0LI
YQkbqaKaO7TRTWxqCJnMZRUlHKSWx7yim+iRISQxgXwWs44SDnGZZjr5wqBR
IYwln1kUsJkqrvKULmLSQkgjh1yWsYkqarnCE7qITrefZKYyjeUUUsRuSrlM
LY08oolOXvOGPn7wk1/8JjYjhDjiSWAIiQxlGEkMZxKTySKbKUxlGtPJIZcI
yaQwglRGMoo00slgNGMI/HGOKHM0MfSjx9pnevnCC57TRjv13OI25znHVraw
ljUsZAETySSRIf/ur6bveEsTNVRQzgbmk00WKfz2Xu9p5hIXqaSYFeSRRn8+
ycBLGrhABeUUs5p55DGOBH7J2Ec6aOABNZRTTAGzGc9APshuKy3UUcl65jOP
LAbQnx690EoL5zjLDraznGUsZSYzyCOBeOKIpVfv9fCZbj7RyEMaeMB97nGX
O9ymnjOc5hQnOcFxjnGUI1RTxWEOcZADVLKfCsrZx17K2EMpu9nFTnZE/v8v
fwH4W6wt
         "]],
        RowBox[{"-", "0.012`"}]],
       Annotation[#, -0.012, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1E1MD3AYB/CfqeQlcrIQ2dhsMetA5aQNcxBbWGzmpTgwQ5hOarhRdGBx
UiY2SuJibKzyVhRS3lJR/UulDg4uCp/m8PmdntP3932eBTmHsg5OCCFEe5oS
QmjmL38Y4zejJM4OYQ5zWctq1rCL3RRzjpu00sYvps8JYQbpbCGPS1zmPi2M
jM/MDWEZ68ilgAtUUscnRpiYGMIsklnFZvZRwEVuUU87P4mbF8IiMtjBTgop
pZoXRIiaLwOSWEk2RzlGKVXc5hkR+ohKCiGB2aSSxla2cZg8jnCGsxRRSRW3
qaaWOup5wlOe0UEnXXzlG9300EuEPvr5zgCDDPGDYV7wnAYaeckrmmimhjvc
5R5lXKGcqxRwgkL2kMteNrKBJSSzlJnE81suX2jnMY+ooJgiDrCBTFJIYEz+
o3RSyw2uU0I+2aQzjxiG/OsgrTzkGsXkk0Mm6SwknlG96aeFB1RQTD7H2U4G
i4kloo+9NFJDCefZTybrSSGG6PH+6vkwjTRQThlXOM0pTrKZTWSRThqprGA5
cUxjKlOYTCyTGLKDgwzwnX76iNBLD9184ytddNLBF9r5zCc+8oH3tNHKO1p4
yxteJ/y/B/8Aaw6zhg==
         "]],
        RowBox[{"-", "0.016`"}]],
       Annotation[#, -0.016, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6656, 6657, 6696, 6695, 6694, 6693, 6692, 6691, 6690, 6689, 
         6688, 6686, 6687, 6730, 6729, 6727, 6728, 6756, 6754, 6755, 6801, 
         6799, 6800, 6827, 6828, 6870, 6871, 6903, 6928, 6929, 6970, 6998, 
         6999, 7026, 7066, 7097, 7098, 7123, 7151, 7184, 7216, 7241, 7268, 
         7295, 7317, 7341, 7363, 7364, 7386, 7407, 7428, 7448, 7468, 7469, 
         7489, 7509, 7530, 7551, 7574, 7596, 7620, 7644, 7667, 7694, 7718, 
         7719, 7746, 7776, 7808, 7832, 7856, 7881, 7882, 7910, 7942, 7969, 
         7970, 7995, 8024, 8025, 8053, 8088, 8089, 8111, 8135, 8136, 8167, 
         8168, 8197, 8198, 8219, 8220, 8249, 8250, 8251, 8282, 8283, 8284, 
         8285, 8307, 8308, 8309, 8310, 8311, 8333, 8334, 8335, 8336, 8337, 
         8338, 8339, 8340, 8341, 8342, 8343, 8344, 8345, 8346, 8347, 8348, 
         8349, 8350, 8313, 8312, 8314, 8315, 8316, 8287, 8286, 8288, 8289, 
         8290, 8268, 8267, 8269, 8233, 8232, 8234, 8207, 8206, 8185, 8184, 
         8186, 8154, 8153, 8125, 8124, 8100, 8099, 8075, 8043, 8042, 8012, 
         8011, 7984, 7960, 7959, 7931, 7901, 7900, 7872, 7845, 7823, 7798, 
         7797, 7766, 7736, 7709, 7685, 7684, 7657, 7635, 7610, 7587, 7564, 
         7542, 7521, 7500, 7480, 7459, 7439, 7419, 7398, 7377, 7353, 7332, 
         7308, 7285, 7284, 7259, 7232, 7207, 7175, 7174, 7139, 7113, 7088, 
         7087, 7054, 7017, 7016, 6987, 6986, 6960, 6959, 6916, 6915, 6892, 
         6891, 6890, 6856, 6855, 6854, 6814, 6813, 6812, 6784, 6783, 6782, 
         6781, 6737, 6736, 6735, 6734, 6733, 6732, 6731, 6708, 6707, 6706, 
         6705, 6704, 6703, 6702, 6701, 6700, 6699, 6698, 6697, 6662, 6661, 
         6660, 6659, 6658, 6656}],
        RowBox[{"-", "0.02`"}]],
       Annotation[#, -0.02, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6757, 6758, 6805, 6804, 6802, 6803, 6831, 6829, 6830, 6874, 
         6872, 6873, 6904, 6905, 6930, 6931, 6971, 7000, 7001, 7027, 7067, 
         7068, 7099, 7124, 7152, 7153, 7185, 7217, 7242, 7269, 7296, 7318, 
         7342, 7365, 7387, 7408, 7429, 7449, 7470, 7490, 7510, 7531, 7552, 
         7575, 7597, 7621, 7645, 7668, 7669, 7695, 7720, 7747, 7777, 7809, 
         7810, 7833, 7857, 7883, 7884, 7911, 7943, 7971, 7972, 7996, 7997, 
         8026, 8054, 8055, 8090, 8091, 8112, 8113, 8137, 8138, 8169, 8170, 
         8171, 8199, 8200, 8201, 8221, 8222, 8223, 8224, 8225, 8252, 8253, 
         8254, 8255, 8256, 8257, 8258, 8259, 8260, 8261, 8262, 8263, 8264, 
         8265, 8266, 8227, 8226, 8228, 8229, 8230, 8231, 8203, 8202, 8204, 
         8205, 8182, 8181, 8183, 8151, 8150, 8152, 8123, 8122, 8098, 8097, 
         8074, 8073, 8041, 8040, 8010, 8009, 7983, 7982, 7958, 7957, 7930, 
         7899, 7898, 7871, 7844, 7843, 7822, 7796, 7765, 7764, 7735, 7708, 
         7683, 7656, 7634, 7633, 7609, 7586, 7563, 7541, 7520, 7499, 7479, 
         7458, 7438, 7418, 7397, 7376, 7352, 7331, 7307, 7306, 7283, 7258, 
         7231, 7206, 7205, 7173, 7138, 7137, 7112, 7086, 7085, 7053, 7052, 
         7015, 7014, 6985, 6984, 6958, 6957, 6956, 6914, 6913, 6889, 6888, 
         6887, 6886, 6853, 6852, 6851, 6850, 6811, 6810, 6809, 6808, 6807, 
         6806, 6780, 6779, 6778, 6777, 6776, 6775, 6774, 6773, 6772, 6771, 
         6770, 6769, 6768, 6767, 6766, 6765, 6764, 6763, 6762, 6761, 6760, 
         6759, 6757}],
        RowBox[{"-", "0.024`"}]],
       Annotation[#, -0.024, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6832, 6833, 6879, 6878, 6877, 6875, 6876, 6908, 6906, 6907, 
         6932, 6933, 6972, 6973, 7002, 7003, 7028, 7029, 7069, 7070, 7100, 
         7125, 7126, 7154, 7186, 7218, 7243, 7244, 7270, 7297, 7319, 7343, 
         7366, 7388, 7409, 7430, 7450, 7471, 7491, 7511, 7532, 7553, 7576, 
         7598, 7599, 7622, 7646, 7670, 7696, 7721, 7722, 7748, 7778, 7811, 
         7812, 7834, 7858, 7859, 7885, 7912, 7913, 7944, 7973, 7974, 7998, 
         7999, 8027, 8028, 8056, 8057, 8058, 8092, 8093, 8114, 8115, 8116, 
         8117, 8139, 8140, 8141, 8142, 8143, 8172, 8173, 8174, 8175, 8176, 
         8177, 8178, 8179, 8180, 8145, 8144, 8146, 8147, 8148, 8149, 8119, 
         8118, 8120, 8121, 8095, 8094, 8096, 8071, 8070, 8072, 8039, 8038, 
         8007, 8006, 8008, 7981, 7980, 7956, 7955, 7929, 7897, 7896, 7870, 
         7869, 7842, 7821, 7820, 7795, 7763, 7762, 7734, 7707, 7682, 7681, 
         7655, 7632, 7608, 7585, 7562, 7561, 7540, 7519, 7498, 7478, 7457, 
         7437, 7417, 7396, 7375, 7374, 7351, 7330, 7305, 7282, 7281, 7257, 
         7230, 7204, 7203, 7172, 7171, 7136, 7111, 7110, 7084, 7083, 7051, 
         7050, 7013, 7012, 7011, 6983, 6982, 6981, 6955, 6954, 6953, 6912, 
         6911, 6910, 6909, 6885, 6884, 6883, 6882, 6881, 6880, 6849, 6848, 
         6847, 6846, 6845, 6844, 6843, 6842, 6841, 6840, 6839, 6838, 6837, 
         6836, 6835, 6834, 6832}],
        RowBox[{"-", "0.028`"}]],
       Annotation[#, -0.028, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6934, 6935, 6976, 6974, 6975, 7006, 7004, 7005, 7032, 7030, 
         7031, 7071, 7072, 7101, 7127, 7128, 7155, 7187, 7188, 7219, 7245, 
         7246, 7271, 7298, 7320, 7344, 7367, 7389, 7410, 7431, 7451, 7472, 
         7492, 7512, 7533, 7554, 7577, 7600, 7623, 7624, 7647, 7671, 7697, 
         7698, 7723, 7749, 7779, 7780, 7813, 7835, 7836, 7860, 7861, 7886, 
         7887, 7914, 7915, 7945, 7946, 7975, 7976, 8000, 8001, 8002, 8029, 
         8030, 8031, 8032, 8059, 8060, 8061, 8062, 8063, 8064, 8065, 8066, 
         8067, 8068, 8069, 8034, 8033, 8035, 8036, 8037, 8004, 8003, 8005, 
         7978, 7977, 7979, 7954, 7953, 7927, 7926, 7928, 7895, 7894, 7868, 
         7867, 7841, 7819, 7818, 7794, 7793, 7761, 7733, 7732, 7706, 7680, 
         7679, 7654, 7631, 7607, 7584, 7583, 7560, 7539, 7518, 7497, 7477, 
         7456, 7436, 7416, 7395, 7373, 7350, 7349, 7329, 7304, 7280, 7279, 
         7256, 7229, 7228, 7202, 7201, 7170, 7169, 7135, 7134, 7109, 7108, 
         7082, 7081, 7049, 7048, 7047, 7010, 7009, 7008, 7007, 6980, 6979, 
         6978, 6977, 6952, 6951, 6950, 6949, 6948, 6947, 6946, 6945, 6944, 
         6943, 6942, 6941, 6940, 6939, 6938, 6937, 6936, 6934}],
        RowBox[{"-", "0.032`"}]],
       Annotation[#, -0.032, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{7033, 7034, 7076, 7075, 7073, 7074, 7104, 7102, 7103, 7129, 
         7130, 7156, 7157, 7189, 7190, 7220, 7247, 7248, 7272, 7299, 7321, 
         7322, 7345, 7368, 7390, 7411, 7432, 7452, 7473, 7493, 7513, 7534, 
         7555, 7578, 7579, 7601, 7625, 7648, 7649, 7672, 7699, 7700, 7724, 
         7750, 7751, 7781, 7782, 7814, 7815, 7837, 7838, 7862, 7863, 7888, 
         7889, 7890, 7916, 7917, 7918, 7919, 7920, 7947, 7948, 7949, 7950, 
         7951, 7952, 7922, 7921, 7923, 7924, 7925, 7892, 7891, 7893, 7865, 
         7864, 7866, 7840, 7839, 7817, 7816, 7792, 7791, 7760, 7759, 7731, 
         7730, 7705, 7678, 7677, 7653, 7630, 7629, 7606, 7582, 7559, 7538, 
         7537, 7517, 7496, 7476, 7455, 7435, 7415, 7414, 7394, 7372, 7348, 
         7328, 7327, 7303, 7278, 7277, 7255, 7254, 7227, 7226, 7200, 7199, 
         7168, 7167, 7133, 7132, 7131, 7107, 7106, 7105, 7080, 7079, 7078, 
         7077, 7046, 7045, 7044, 7043, 7042, 7041, 7040, 7039, 7038, 7037, 
         7036, 7035, 7033}],
        RowBox[{"-", "0.036000000000000004`"}]],
       Annotation[#, -0.036000000000000004`, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{7158, 7159, 7193, 7191, 7192, 7223, 7221, 7222, 7249, 7250, 
         7273, 7274, 7300, 7323, 7324, 7346, 7369, 7370, 7391, 7412, 7433, 
         7453, 7474, 7494, 7514, 7515, 7535, 7556, 7580, 7602, 7603, 7626, 
         7627, 7650, 7673, 7674, 7701, 7702, 7725, 7726, 7752, 7753, 7754, 
         7755, 7783, 7784, 7785, 7786, 7787, 7788, 7789, 7790, 7757, 7756, 
         7758, 7728, 7727, 7729, 7704, 7703, 7676, 7675, 7652, 7651, 7628, 
         7605, 7604, 7581, 7558, 7557, 7536, 7516, 7495, 7475, 7454, 7434, 
         7413, 7393, 7392, 7371, 7347, 7326, 7325, 7302, 7301, 7276, 7275, 
         7253, 7252, 7251, 7225, 7224, 7198, 7197, 7196, 7195, 7194, 7166, 
         7165, 7164, 7163, 7162, 7161, 7160, 7158}],
        RowBox[{"-", "0.04`"}]],
       Annotation[#, -0.04, "Tooltip"]& ]}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  PlotRange->{{1., 80.}, {1., 80.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 183},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->43200405]
}, Open  ]],

Cell[TextData[{
 "If you wanted to solve several Poisson equations with the same geometries \
but different right-hand functions, you could compute a factorization of the \
matrix and use it repeatedly. The way that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " allows you to repeat the factorization is described under \"",
 ButtonBox["Saving the Factorization",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixComputations#1974718471"],
 "\"."
}], "Text",
 CellID->283159489],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"factors", " ", "=", " ", 
  RowBox[{"LinearSolve", "[", "mat", "]"}]}]], "Input",
 CellLabel->"In[11]:=",
 CellID->237769871],

Cell[BoxData[
 TagBox[
  RowBox[{"LinearSolveFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"6400", ",", "6400"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{258, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->56876998]
}, Open  ]],

Cell[TextData[{
 "Now you just need to give it a vector representing the function ",
 Cell[BoxData[
  FormBox[
   StyleBox[
    RowBox[{"Cos", "[", 
     RowBox[{
      RowBox[{"6", 
       SuperscriptBox["x", "2"]}], "+", 
      RowBox[{"3", 
       SuperscriptBox["y", "2"]}]}], "]"}], "MR"], TraditionalForm]]],
 "on the grid."
}], "Text",
 CellID->1972304745],

Cell[BoxData[{
 RowBox[{
  RowBox[{"cvec", " ", "=", " ", 
   RowBox[{"Flatten", "[", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"Cos", "[", 
       RowBox[{
        RowBox[{"6", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"hStep", " ", "i"}], ")"}], "2"]}], "+", 
        RowBox[{"3", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"hStep", " ", "j"}], ")"}], "2"]}]}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "num"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"j", ",", "num"}], "}"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sol", "=", 
   RowBox[{"Partition", "[", 
    RowBox[{
     RowBox[{"factors", "[", "cvec", "]"}], ",", "num"}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[12]:=",
 CellID->2115757770],

Cell["\<\
This passes the vector to the factorization, thereby finding the solution.\
\>", "Text",
 CellID->1506613364],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListContourPlot", "[", 
  RowBox[{"sol", ",", 
   RowBox[{"ColorFunction", "\[Rule]", "Hue"}]}], "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->671126971],

Cell[BoxData[
 GraphicsBox[GraphicsComplexBox[CompressedData["
1:eJx03Xd4VHX6/nFkLVEQokQMGjQMMYYYwpTMTDKZSR5CCAkQmt3VtawNe+/l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   "], {{
     {Hue[0.], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{6766, 6793, 1465}, {6791, 6764, 1450}, {6544, 6543, 
         654}, {6579, 6580, 747}, {6576, 6575, 732}, {6698, 6718, 1228}, {
         6819, 6848, 1622}, {6870, 6845, 1692}, {6847, 6876, 1700}, {6908, 
         6872, 1774}, {6794, 6820, 1544}, {6739, 6765, 1386}, {6719, 6738, 
         1307}, {6595, 6594, 811}, {6596, 6597, 828}, {6555, 6556, 665}, {
         6630, 6655, 989}, {6629, 6630, 989}, {6874, 6911, 1777}, {6613, 6612,
          890}, {6844, 6817, 1611}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJwtl/mbzmUUxl/vlssSouX9UhJZR4iQ6i8o21A/SJqUlG1KIUOEbKMrjL2F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         "],
         VertexColors->None], 
        PolygonBox[{{6612, 6595, 811, 891, 890}, {6792, 6791, 1450, 1451, 
         1531}, {6556, 6578, 746, 745, 665}, {6655, 6656, 1068, 988, 989}, {
         6614, 6629, 989, 988, 908}, {6718, 6719, 1307, 1227, 1228}, {6594, 
         6576, 732, 812, 811}, {6738, 6739, 1386, 1306, 1307}, {6580, 6596, 
         828, 827, 747}, {6818, 6819, 1622, 1542, 1543}, {6845, 6844, 1611, 
         1612, 1692}, {6577, 6544, 654, 734, 733}, {6873, 6874, 1777, 1697, 
         1698}, {6793, 6794, 1544, 1464, 1465}, {6872, 6871, 1693, 1694, 
         1774}, {6846, 6847, 1700, 1620, 1621}, {6765, 6766, 1465, 1385, 
         1386}},
         VertexColors->None]}]}, 
     {Hue[0.08611832769144702], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{6822, 6849, 1629}, {6544, 6577, 653}, {6629, 6614, 
         909}, {6460, 6459, 333}, {6696, 6695, 1205}, {6844, 6845, 1691}, {
         6495, 6496, 431}, {6509, 6508, 488}, {6574, 6573, 726}, {6657, 6677, 
         1075}, {6719, 6718, 1308}, {6595, 6612, 810}, {6715, 6696, 1205}, {
         6596, 6580, 748}, {6699, 6720, 1234}, {6576, 6594, 731}, {6874, 6873,
          1778}, {6944, 6973, 1943}, {6598, 6599, 836}, {6767, 6795, 1472}, {
         7027, 6997, 2090}, {6557, 6558, 675}, {6656, 6655, 1069}, {6721, 
         6740, 1313}, {6871, 6872, 1773}, {6850, 6877, 1708}, {6766, 6765, 
         1466}, {6615, 6631, 916}, {6999, 7036, 2099}, {6791, 6792, 1530}, {
         6578, 6556, 666}, {6512, 6513, 513}, {6796, 6823, 1551}, {6523, 6522,
          567}, {6819, 6818, 1623}, {6878, 6914, 1787}, {6488, 6487, 410}, {
         6473, 6474, 346}, {6868, 6843, 1686}, {6794, 6793, 1545}, {6913, 
         6945, 1865}, {6739, 6738, 1387}, {6847, 6846, 1701}, {6524, 6525, 
         594}, {6972, 7000, 2021}, {6995, 6970, 2008}, {6941, 6907, 1847}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmFlwlmcVx9+8Sb4vtC4z6lDfLwrlhlqQrb1gSQhZIAQse4HQUp3xossU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         "],
         VertexColors->None], 
        PolygonBox[{{6490, 6460, 333, 413, 412}, {6793, 6766, 1466, 1546, 
         1545}, {6581, 6598, 836, 835, 755}, {6997, 6996, 2009, 2010, 2090}, {
         6522, 6509, 488, 568, 567}, {6912, 6913, 1865, 1785, 1786}, {6543, 
         6544, 653, 573, 574}, {6510, 6488, 410, 490, 489}, {6513, 6524, 594, 
         593, 513}, {6765, 6739, 1387, 1467, 1466}, {6631, 6632, 995, 915, 
         916}, {6738, 6719, 1308, 1388, 1387}, {6764, 6791, 1530, 1529, 
         1449}, {6740, 6741, 1392, 1312, 1313}, {6630, 6629, 909, 910, 990}, {
         6911, 6874, 1778, 1858, 1857}, {6998, 6999, 2099, 2019, 2020}, {6872,
          6908, 1854, 1853, 1773}, {6474, 6491, 427, 426, 346}, {6943, 6944, 
         1943, 1863, 1864}, {6848, 6819, 1623, 1703, 1702}, {6575, 6576, 731, 
         651, 652}, {6655, 6630, 990, 1070, 1069}, {6677, 6678, 1154, 1074, 
         1075}, {6817, 6844, 1691, 1690, 1610}, {6845, 6870, 1772, 1771, 
         1691}, {6720, 6721, 1313, 1233, 1234}, {6597, 6596, 748, 749, 829}, {
         6594, 6595, 810, 730, 731}, {6718, 6698, 1229, 1309, 1308}, {6496, 
         6511, 512, 511, 431}, {6821, 6822, 1629, 1549, 1550}, {6869, 6868, 
         1686, 1687, 1767}, {6695, 6674, 1126, 1206, 1205}, {6580, 6579, 667, 
         668, 748}, {6612, 6613, 889, 809, 810}, {6573, 6542, 647, 727, 
         726}, {6849, 6850, 1708, 1628, 1629}, {6795, 6796, 1551, 1471, 
         1472}, {6877, 6878, 1787, 1707, 1708}, {6942, 6941, 1847, 1848, 
         1928}, {6876, 6847, 1701, 1781, 1780}, {6820, 6794, 1545, 1625, 
         1624}, {6556, 6555, 585, 586, 666}, {6525, 6557, 675, 674, 594}, {
         6716, 6715, 1205, 1206, 1286}, {6971, 6972, 2021, 1941, 1942}},
         VertexColors->None]}]}, 
     {Hue[0.18463086745403604`], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{6999, 6998, 2100}, {6721, 6720, 1314}, {8168, 8167, 
         5503}, {6880, 6915, 1792}, {7191, 7158, 2489}, {8047, 8046, 5191}, {
         8049, 8050, 5193}, {6476, 6477, 360}, {8110, 8109, 5346}, {6497, 
         6498, 441}, {6868, 6869, 1766}, {6878, 6877, 1788}, {8255, 8277, 
         5831}, {8190, 8189, 5582}, {7155, 7126, 2406}, {6822, 6821, 1630}, {
         6633, 6658, 1001}, {6491, 6474, 347}, {8306, 8294, 5983}, {7096, 
         7130, 2340}, {6913, 6912, 1866}, {6632, 6631, 996}, {8222, 8238, 
         5673}, {7128, 7164, 2417}, {8239, 8254, 5752}, {8083, 8082, 5268}, {
         6432, 6433, 195}, {8253, 8252, 5819}, {6916, 6948, 1871}, {6769, 
         6797, 1476}, {6458, 6457, 324}, {6542, 6573, 646}, {7125, 7094, 
         2325}, {6996, 6997, 2089}, {8273, 8253, 5819}, {7038, 7065, 2184}, {
         7061, 7026, 2164}, {7064, 7097, 2262}, {6941, 6942, 1927}, {6944, 
         6943, 1944}, {6507, 6506, 483}, {8140, 8139, 5425}, {8276, 8300, 
         5909}, {8086, 8087, 5274}, {6511, 6496, 432}, {6743, 6768, 1397}, {
         8296, 8307, 5984}, {6436, 6435, 245}, {6514, 6515, 522}, {8291, 8274,
          5900}, {6798, 6824, 1555}, {6741, 6740, 1393}, {8191, 8221, 5594}, {
         6715, 6716, 1285}, {6947, 6974, 1949}, {6850, 6849, 1709}, {6460, 
         6490, 332}, {6796, 6795, 1552}, {7002, 7039, 2106}, {6404, 6403, 
         167}, {6488, 6510, 409}, {7160, 7194, 2492}, {6524, 6513, 514}, {
         6825, 6851, 1634}, {6700, 6722, 1239}, {8220, 8219, 5661}, {6723, 
         6742, 1318}, {8237, 8236, 5740}, {6440, 6441, 278}, {6674, 6695, 
         1125}, {6598, 6581, 756}, {6509, 6522, 487}, {6678, 6677, 1155}, {
         6968, 6940, 1923}, {6582, 6600, 762}, {6972, 6971, 2022}, {6659, 
         6679, 1080}, {6852, 6879, 1713}, {6975, 7003, 2028}, {6557, 6525, 
         595}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNWVl0ldUV/u/NzR0UGXNv+O8N3EDC4GpfKhkUayeZFFq7FFAcAEFFARNC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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1GtM1lUYAPDD5X3LaqXp+xovoYA3WLfVzEBJu7mGYi0htdoCWSI1CCFd
ZdplGLR1+ZKAuOxTy/ogaWtLp1+M1FqtPiRQbV2WCFh9DO0i1e/d+fDbzvP/
n+c55znn/74ljW1rn8wNIdSQ4LOCED6nOi+Ev7z4m3WZENazzYQFLKTcnNVs
zgmhiTSrsnlcZnw5j7Bd/BxL1BkTj1NZGEIFnZeEsIvTnDLnJH+YN8lK679q
zTaOW29GMoTpXF8cwrUUpUNopYXb1SpmjzoTjPO2WvuoVauOd3jInIdZemkI
v5rzGwfMGWDI+5fpImXNHBbn2x+d3CW+OxN7HxaPsEhcxkfGj7GJ2eJr6Da+
jQqWqV9Fjd4vOuOpEPOWkVAvPxFr7Be/lol9NlBPoV7/nRvCV6kQ5qRj3+1q
beGQeh+Sa8+N4o7s2RtP44jnS/VaS5l+FzGi3x3i9SwW76KTneLnecW4gkrW
WvcBjlrvGFfMlqf+BvrVrs7eNwvE+ezX0wXO84X9DvDLrBBuse+bqdPXnZTq
tYN2NqvfTIPa05lBlfiG4niv2XWz6z+h/jp2W2+UM2xRq51P1dmaiN9mrbw6
UuqckHeSD+zjTXbbyyhnaDJnE1eadxX1lOg/h1XOaC/99GW/KYadRxfdfCw+
zPxpvlvxEB3q97JX7X6qOW5v7/KoOxzgAJV6mM+kM1qph3t43Zw3mOn9YCL2
s0H9KsbsYYn6t/Kg+Cm2Gz9LOe/Je59GuVcnY40ifc2hXO/fOffv+cecIrnd
6nVlazKYPTsOydmYjDXuEF/HN77FeYl4V8PyhkjpN00TveIettrDNk6wRl8b
C+J3d5hmdop3FMTvMy83/hZLxAuZ5f1MBp1FrjXzMvG3MuF3f47l6q/gd3t9
i300yKvPktfAQY7KO82Y/Z+lh5/EPxPkHEzGHnPV+tGzi8Y1nq/mgmd/ckxP
LbRyXjxJs/HjHOEHeZ9k4h1OJWONc+77y7nx236JFxmR9y3tckooJU+cz/3W
+y8Z9zQlb4IX5Hydir/tUnPmFcY+13AfPea0ca/vqY9ebnS3N/GM+y1Lx3te
Ll7BKeOn0/HduHMaocdZnmWUFnFrQfwP7mMP/wML2Muj
         "],
         VertexColors->None]}]}, 
     {Hue[0.283143407216625], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV1Gto13UUx/Gj/rWLj7S1m9PUcunILUi0eVtl+GDSZltoClpT12abTv/V
vNAuYqaiOWvMhFRSULyxogcqWlC4ZGv1aNhlLsorGkQEeUHz8vo/ePM5nO85
53u+5/v7/kYtqitb3j8iajEAnw2L+Bp9D0c0jYxoxHOZEe/TBqSzS5+IKME/
j0XcGBFxHQn2VuttaBdzJjviIo4NisgQm44yMU+qPRp7Hop4nm8S9vK/J68e
pzMiVtNVuMIenRWRhx69JegYHGB/Yv1j/GSvLr5OFDtACV2MZL+Ip2gh/hM/
h76NVv4N8j5EtdxNdCOa2UV0OjrTI7bp/SOk6f+oHo+gN82M+PqjMBFx1jl+
RuUjEb/SX5BkN9IGbDbDE/Q4xvD/Ke8P9HP263w3cIL/Fr2Jb9gJOgCzxWSK
zcAme223fwu26KGD7zS+0lsXXyfGPm6e8grRI7eITsff7FnOPQ4Zzj5X3hzU
D3Tn8mowTc0ZfC/haf4dfEkU85eaRwn+N5NXaRkGu5cWuh2nzC0hLwvT9Pk6
TWKtOhXW30Sx+Hf4WtDFn03HY6v4z/V1HF/6CE/p93fkmUO+9ZdxTswP1n9D
qfvtpX9hD7tS7bewUP1BqRmj3HkvyBvITrCPsL9HlVm9K68JQ+yVa/0F7BPz
Il2If9nr6C5sc3fValehQv0FfM0o4m81my/Qaj7ttBvn2T/Sa3jWXUzIcYeo
eDRishqFqXdhhn1q3MdqZ7xHh4s5w/6A7sfMwRFL6UbcljtC3nDkyc2j41Iz
ZZdYr0W7mKQzLcJJ91tgPR9rxLxGy1NvVv/39DUs9V70dpV9F43sOtqGmfrP
UW8i0tRcL3YL3nC/4+lU1KiZQ59BDns5XYaV6q+gddjJ3u1/sAvZQ30jZn8Y
mebfzNeE7iHmot9XsFTPh6wfxGIxWWpkYqj6d/R1Gx363GlWn6bemPmvFLsi
da++pfl88zCF/7LYS6gXn6v2KHR4+7PsWYwC+/bRczimtzZ5O9Aht5JvCRrE
fEe/RbWYB8wIuXk=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNW3d4VkX2vumNUNLuvd/3kQRcAQELuu6u0gTR3UUIEFogEGpooaPSmy66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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwl1WmQz2UcAPCHvRw1xd7/VVaOWEozXZKlol64o5gpYdeRm93VOmawaJqw
qyhyRGmqqRS6tsX0AmsjTI2mkq0JMyFNh9HQgfr853nxmX2+v+f7PL/vc/z+
26505rAZTUMI5aSRkh9CKm9qj2Akq9uG8EJSVgh96Ud5IoQKDho0j/k8IO7G
sdQQOotv5j3jtnHSmFWsZl5hCLNYlxvCWc6Qb1yCamN7UUw/8YMUmee4uJFS
44ZwOSeEgcYNYJ+ctxmXHsKHfEBjQQjfM7e5WujCLXJuTcR5VqTGd51R21kW
ZuujO6vE0yhWaw3VDBX3YEtmCOf5g0Piw9xmzImsuMYr4qsc8WxBdpz3Oe98
njx1HUiLe/aWeCVZno1Pj7WvV+9CljUL4TuOMUJO30TcyzlUsl/cQK0xE3mS
Xcbtpsg6jxvXyHR1DOF3tfZRW29Oiw/SWV1zmUONucrYb+7s9FhTrbm+ZoL5
xpPFKPET9DH37xn2gI7iTrymPYzh3Ce+n/PaWzJiXxtxD77SPsoSTnrnKVK1
P/LOj0mTk14Q50nJiH0J5/2vmuvVmuXcM9kgZ2PyjNVSTQ13y8llmPW+lhnP
qsLYSZS4J++zg8/lvMtx+1Fkzi70ltOdeeY+yAEu3Wh+aluryVxpzFLvyEQ8
h4a0uGfnfC+/8GtKCJvYzHfi4/mxPZghVKl3MSvV2rtZ3MdP5NTxn49wOzso
l7OGev37qOAd79yaiGedmx7v0mVr+IlK9Tewn1w5Kdzjbi/jWZZY11LG5IVw
HddTIG6TZJ2Hs+N9XdomhKd5qGUIF1tYPx3UkcpQ5/Aqr9Bb3J7N2r/xK/dY
Q0/6Ngnhgt+MP1kk5zHuVf8KlrNIThVjrfN6WjFIPJA8Y/OZzFC19WSb+q5y
ha3W+jwrnFtjVjy/G+T851mj+rvKKaKduQqpUMNs6tmpjnV8q4ab3eVOye9E
XEdH7aeYzSTrn8w/1v4OW0mRc8KeNrHOhzPiXjQV/+wd55xrz9S431PEUzml
/tW8QJ7aLrpDqe7OIwynSE5XbtJ/VG1f8aJ1PEuVs/mSL+gipzMN+j8jV/5l
tRyjXK172cOPaqlPxN+9phmxzuXmWsTt5lnKErqK82ltj0opocL85ewy925a
Oo9ScUl+PIcpTWLfAXEt/+ofYM39GScezx79ZZRTYO/akGXf6tW2n4vqzfWs
uWfNqPMsIf7L80+1r/XsGqarbTZ7faN7KKNSPJ8s9Weyz7Np4kfI0Z7IBHqJ
i9llTVOYykRxOdv1b2M0w8TjGKX9OC2Tf5O/aQXxHhe7H72S91VOdfL+eefo
vPjtrBGv5Yr2e3mx7xlxDTu165hEN/EthbGGgpz4jS0QP82T+ieSw173YhMJ
vy9TmExZ8reK/urawHpellPFoVbmax3zk2t9uDDW3yI3rru9s+iQH7+913mD
xXLmclrOXd55JzPcyZk85Pu56D5e4l45RYw1zw62J785e3KVBmdUyRwWq2MC
C9RyhMPcatyNhfHbm58Tf0MfFfcpjL+jed6dmzwr4wbQ3ZhFLKRcHWX8rYZ/
6K+mH+TsYZJ11vEJ36jjArVq+Lh5/L90uzt0ByXuTita85K51raN8wxgI4Pk
DGab/tGM4X9p3UBB
         "],
         VertexColors->None]}]}, 
     {Hue[0.38165594697921396`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwl1VlslWUQxvEptUBZVIRKN4FGK4QGGmiN1XIhshSila1qNC60KLRRQBFQ
DF3BrbZhCSiLoNhEZTH1BlE0IEtBi8qFibECsSVColRMAMUERX8nXvwzX+ad
5Zn53u+cnIqFMxf0iIhmJONidkS/myJe6RdxMiOiGy0OaoZEVKNtQER+ZsRk
dF4TcTUr4h8cTY04wteGPT0jfpOX4vmC3K89n0CZ55NiT2CZ+GHDInJQdGPE
Pr5O5POn8Q1CKv+VoRGZng+lRVToX46lNLzAV48zgyPeYNfjanrEnex0fCi3
lm3EWP46/WtRbthX+dahln+EnnfjvV4Rz/Atw5tq1tlBLR67NmKnOXag0lxT
9Z+CETRkiM3DAL3q+N5CO/8OfZpQo9dc9nm0J0V8wu7J+H/Rz6m3CO0pEQXy
SnH6err5juIpve41+z34ZaB9sANR6rmerg2YQ9t42u/CBfoX0zIfi+ifpU8R
ivW9lS8XD9C5Vu1dyFZ/pXe9AiV9I86J/RXnadtCy0Fk3CBX7Vy0qL+V3Y+u
3u4LLU14aVDE+2ocwFZ1qvgqcTv/o2IfSegT/6R69VhuP0/oMwcHaRtnjllY
3N+szuditJje8objQX1XsFuwVp2f2B56Vbsn2WwW0vpEfKXnWQx3T3qr0QvF
9lli1oeQb89NdvA6Wr33ofLuQKbcQr4CbLOfevXrsEqvJOdDcEyvZucvo0ru
GX3+xhK9RpljJvLN8rA+i1GjV082C+N9I5XO70ezmF3y2tFhPzPVXojdNJwW
24UU8xbxlaGSf0zi/ibuMW171TiW+K68o3NqdGM9DZvMuhEDva/97D5UeJ7m
fD7OuzP3qVeKVjXz1BiJm+kZreconKRzpT7VGGzGzXawCafsIfiScFavz/j2
Is9OyvmexVT37Xfv8Dy6r4uY7bwBE+ROYMvxh7l+1LMDb9M/TV4FrpgrU/8M
NNOwgq8Bs2k4ZI7DWGSWp82xBuPs7Qjbhhx6XjPTdsxw995V+1N87Ls7o16q
vqn6LnDeiGQxXfI6sUDuanYnGtX8lj2OifyFYgtQYVe38RVisx2OpKsYj9P8
F99lfC5+iV5r8I33Pl3/eVhut33tow+O+y390xzp4pPV6VD7UuL+8X/BnkA3
bQfEnMI8824X+wG+o60qcRdwhZ4O5z/gIzFFeuai1d6+ZH/GXt/1Bnkb0SJ3
cOJbxAx9JzmfiDw6c+kqRq7vro09jFLvbint9Ui3wxfNuhSHvd/+zm/B94n/
Bf0vYzcNk5xPRqed9LTvFJTZeYMa2xK7UKdI/+l4h4ZL8i7iX7nZbBYafadT
1CjBOXXG8o1BB/9qelcl7oe5/gNEDPTj
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJw1Wgd4lVW2/W8iBAgl9f73khtyKaElhBYInRBERQkQZJRRR6QLltBCgIDS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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1mlwldUZAOATyIZCWUNykxAMAqIStrEkENoBdQSXJIBotTMCBkECiAoB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         "],
         VertexColors->None]}]}, 
     {Hue[0.48016848674180296`], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV1XmQz3Ucx/HPlj2tMzJr3RW6prUqi6aMotXYYrEVNUTMkqPU6hpMm0yb
Y6ZkW+SoWXZYx8iMppZ1FsksmqnQYohhQmsxrpmOx+eP57w/+/68P6/38fl+
fttx9JT8ybeFEMpxOx5pH0IeFt8BdgkqWoSQ3yaEwdicFsL77UJ4D9ubhbAi
M4SdOJMSwtCMEIZhbkII452bj0ed7dMhhGzMuTOEi/Yv4Ipk09giZChgFL2R
mEozhV4yXkgOYQw7G7n0t9k/ileah/BZ6xA+RdukEL6RZxNuyPWR2OUoFX+k
bQiHsahJCAlqD9ifGsKzNPKQLddO53Ygo2UI89g1mEWnht2Iq9Yb5PkJU+Tq
R3s0bqqtG18uTjcIoUB//XFSjy3ZrkixzpGzB8ab2yF6v+AZuc6zB9DPeiz7
KrrJVUTvLdQkhrCab088w38m5kGavMf4ajHO2QFsLk67r5fsT0Oxs3V8l7BU
zGW2PvZjvd481uFW4xBOmP1BvOYujvL9jS/NqqH+7sUIuR60/wCy3Gk+jSFI
Uk8Pvhz04r9P7FNYI76W7w+U03zYDPKwwhy60H4C9zfyXbEbsVkNlWbzA1Y2
DKGx2m85m26eM9zPEux2RxPEzkRj8aPkGRnzud8S5z5GvrOT7BejlZgy575A
E9/JYrYaza3nil2L58WX6ePN2I9eGqoxDTfMJ4VNxVXrS2LrMD3dN0B7N/6h
X8+XFmviP2J9GL2s28jTDSVNfQtsP9yl/gKxE7FX74H2BXlL6WdZd0RP80nS
eyL66v0KvcuxN5q72OPxbzWPpjcdRTQb0UvH72Jq2XqsdHd7aexBlW91i9gq
vKz3OvvJ1ufF7GKXoRN/e/kTcFY91+RpKm4+zRr3cAzD3eMAsWPQRd5sNheH
9dhEbGfUxt7lzIxvUv338N0d9/TbgGYnHPTujrHX0Uf8dnYbLuprnXNrMVHN
v9H+FZVqq7C/CsvFpPBlope8n5hfCWa6u5PsTUxS/+NyDsO78s5S+yq87rel
d7xzTPIGJ0eLBjQHix2ELeIHiX0O033PSfIkoodcP9L+Ex3ol4pdiErfQHZ8
r/EevbvO5peDYe7xbX0swCFvcKjYIfG7oT9Q7AScE18oz/g4X7lay5OB2XKd
8l0nqq3YrHLpjcRfNG8418q6Sg2L6JVhkxqynHsozsvZHDl74j/zr6azFVl0
rtpv6fy/5rnP/jlUm/PTbH9kqbNdtFjo7HF1XcJq9342/t7QSfeum9FoGn9r
1H+E5hVsoPm92O+QJP4yXz1u8RfoYyqqzLw92w6Z1h/SK8YC7/dJti+u0X/D
/jykupc6eslyX6e5NdaCfe6xRfx/goFqqOCbg3f4v2a/wrfWKfF/QuxNL1v4
DsTfEzpr2ErUiClmP0Ch9Up65djvWxru3IuYaSYD2BHobt0xvu/Yg/o/51uP
DmZYzBZiqZjH3ElvFLnri+wFnPPWxskzFj+763K2Clvl3aHvk+jujk6wxzHD
ehm7FKfMp7W6MjFWv135uqDCrPZHvfi/Vl//AwMdIuQ=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNWnl41FWW/VHZScheNV8llapqNpd2FBIWoRVUQHZoCKDYouOI7AGn1dFW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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1mlwVtUZAODDlpUQlmAnX8hHUBa100oIthDUdhRCQDtD2VpRKlBLoWy2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         "],
         VertexColors->None]}]}, 
     {Hue[0.5786810265043919], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV1HlM13Ucx/HPREFOU3IhqShaLU8KS9N0qQmhtTUXphOPLGfrQFkmZc2a
QIe26Sag00w7VSi8WkvFVLQlmU3Jo1Z4pOm00MSBR2v1+P7x3Pvze3/en9f7
+Hx+354zZo8vbBNCWI0Y1HYJYQeS/djbLYSfkdchhKzuIeTiyC0hrMwIYS6e
vDWETvb74PekENLtD0SZmIl8hTiYHMJHt4fwLS60DyGzRwjxaOkcQqzYLrhf
fB69WTiTGkIFXzkSOoXwFN909JXrWNcQLqO/XHnpITyCQe1CaEPvbzEf0Lys
9ku4rv4xcj6Mm3HqY0+jJD6ECerKR73aLstzCS1y/cQ2YJ316kgPNfImyXOD
ZlzbEJrZRL//o9+qlhYsUU81W4XPEkP4k/ZFnDW3PnLejWo1jKc3FF/pMZNG
MibSHCF2OOarpwObgkaaE+jlozbBTJ0rRbF6mtlUPW/SbzaNsbhAZ488uzFI
j+/T2IrtKfKzDyGL/hZnV2E9nc/5qnFNzCG+rbjB355eHHJpHjOPK6gykwFs
fxx2X6/ar8BR86/lO44CMVvobUas3t9T+2JMNJO1tNdgL/3hkQ9F+hoidjCe
U9v39OqxK9YM2A4oUEMluwm9+cv0V4rl3lKznJ2imuUtojcHCXKl8d+Ghepc
7742oN4jX0ijBE1q3kfjDAabVaJZ9sY18zxF4yZyzP+HaK6ok3egPAOw1N0d
tX8EfcXcYz8Lf6izlT2EX8VPlX8KCjvSFjsCvfS72f4BvCSmUd9Xsd6s1qpl
Dc7pa5E867DArL5mtyFGbV+K3Y948TP5FiNLzCf6+xjf6HEmjXfxKJ2b8v+D
NDUn8iWgwDsc5tyDkS5/E9uAcfTraO9BDv1P+fbhAP0C9U6OZmduTfbb0Twv
JpleX0yjOc7+FAwRs93+NkxyLxv5ajBHv5PM4GXsNod7aY9FhRrO0ojlf8td
xLF34Kj1vOi9o0ANo+mNQoZ59hOfgxp5b4hNVVO5u9gZzQDp/pvH+Y6hQp0v
si8gzTt/lp0VzdD6LvZO9PP2uquxGyrd4wp5diJF/W3ZGGR7S6+xy7FLDY3q
asVMczsp56nouylvevQexXTWV471GJywblTnbxipzkViF6PYfSXbT8IoMcPs
D0Wh+byjvw+x0j3W2T+JNLlOONeIKrlqxO6Pvjve/EW1p4jvaCbl1svQ08w7
OpeNXPptxcbgsNk26/sKWsxnunpfR5G+RvJNwhtm8gD7eDQ761zaz+Bf+k/Q
y0eCe0mI8uAxcytm56HBG1giz0YUqO0XGn9hhVw58kxDplw7aGxHsr7edG4W
VjrbI/quYxXNV8QWo1Z8GY1VeN7d9aTdAxlmtZetw1W5pjo3H2/TuY/thQ10
ZjtXiq7OVrJfoMq6hF0YvQnrH831ICab7Xf0TuM6zQXRO8JkMXP5liKF/zzt
c9F33EyeNpsZyHNfpTRKUOR+y51bFv3Hnf0fgZkX1w==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNWWl0lNUZ/pp9ZpJJmMwIM5mZbwBpeypRrARsxfYUWWQT3AhZWESW4MYi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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1nd01dUdAPDfacbL4GVAUHiPJAQ8XYCgMnqqHSoEEO2wVQgJuCsHLUPb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         "],
         VertexColors->None]}]}, 
     {Hue[0.6771935662669809], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV1HtMkFUYx/HjJQSEUJDKC4hp0wTMbGItChwqS8tas8klnFbq3BLLFHVL
0y0Tq+nEy0oYpXbXMgMrpJW3iXnBNpVqdlFplqizC5k0bX3eP757zn7Pc37n
OZf3HfDEnEfLO4cQqtAlimkhrMHbN4YwQ5yOGOMF/UJYi5TuIRTRJqM5MYR8
2uOoiA/htPgzChNCaBIPoEP98fQQfsc7ySGURrXYrz6hTwiJmNw1hDjaEHwf
F8JgMQ+D1cyxTjnS9JDYP4ThGJ0SQjf5GLSobxWv40HrDhIHYpC5lX1DWIHq
2BDuMK8Qq839SP5DbNVbDu+RmGkv8XqJw0T93E4bgjvpGbT+qKGPpuVjKf0B
3jPQiX8yrSda9bBI7UK03BDCBVoXe29NcjbGf2Ob8f16ycNu/UxVuxir1LeJ
55HczbnJN6BragjfmteGdeYWiCV4UQ8PR72jwficXmLt62VnspTHMrTxrOGx
AGW9QlhCm4H19FvFgXjPvjrMu4r1+u8W3TkOGu/S+wkUubsm/r8il38av3SU
6z9VvheW9ODRmxdieI6TfxI/qikSi5Gihz/l/0BQ82Z0BthL3897H+7mv1nc
hDZnW8V7DWL0kC/moW/PELLETBy1brt+/8JV51NBm4/P1XQWO2G4mk3i5mgO
n8/ETzHJ+JjYjE3GcXqJRa6er9GuI824VCzBLJ7nrfMbzljrIu0CLptbqN9x
+M/d7Zb/CgVqToonUGW8w3l/jHkxIdTT6hBP/5fHaRzk8wHtfbR78zViNRqN
l0d3jqeNz6g9iyPqd7i3w9juPReLRdjj/R8Sz0bfBn2e+BySjDda/wWc8wZW
01ahjD6F3yLMtscaWjVW0mfa06t4xF0UyJchXc0wcQy+cbbD5Mejzt5/EVuj
b8893mWdEdE7dNeH9P4Tyux3uvxTeIjnfdbJxVw9L+S3AY38v6CdwCk9LHcn
W1DhnayXb0Ccvb/F4wAuRe9E7SeYpn6a2koMVT9bfiJ6q9kQvX3c5vxfoW1B
Lb2HmB31pP/X5OsxVM3X0XeAYj2XinNx3Ld2RP4w9qlZR9uKd93LNevfZLzR
t5PFLxPbeU5RW4ZKZ54lPwY5fLrax80Y4Qzn6/cxdNfzTrXVGMV/hdo38Az/
CbTxuIfPRTGJfyr/RDEBU42fVfsSBqhPFbOj96qf0bzzcdJ7bjb3Epr432L9
bCzTQ718LbbrYa95LRhr7wnyGZigZjFtLYr4X+HxDzKt22jeKST4Xz1Pex0/
6HOkmIMOb+AyjyQ17e6unXYM3+lhAL9R6ONMMqN7w0p93iv2wyzjWvmd+NK6
pbQSZOhzUvS9YpeaOn5Ho/8vz21q9+CK+v8BGVwCRg==
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmXl0lsUVxl+V7AkhyYdHvu01KhCwtmrRsm+iQrCCVEAjyKoQlqCySIIL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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1neQVdUZAPCjwa1sfZDI230biIJxF0hsEREpYgWDTkapWYqCCJESpRkd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         "],
         VertexColors->None]}]}, 
     {Hue[0.7757061060295699], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV1HtMl2UYxvFbCJXwlICUBSp/KIan7LSWDA/NaqVLmx0krZFQ1tLMM1Ku
ObN0AyVbLVbKpOPW5gGQ2jptmYBQK621GqUWICusZdYfkfZ5//jufnc913M9
9/287+83rnTlwhUpEbEFqei/KuJf1A6JeGh0xIOoTov4JS+iC8dHRmzLjXgO
q4ZFNKmNGDY84kvrfdjLc/uVEbchZXBEDa0ZafTXeV/Dx/aWyS7HbvklzlyM
I5dG/Eb7FaMGRVSqm9DJc8D6/oSMiFlj9IYTmRF1srfih8siCmm34Hn6RFoB
WkbIVDPoZ/WwXsY6ZMmp0cs7qNdPC08rnuV5gbYXa+nVtCoE/Sfaj3jPvNlq
Fnrd1VGzfo6i9IhX9VuLGwZGTKPdgWaz7OStRp3MGufvQqkeNtEqsJB+yjlH
UemsIv3mYaVZDtO+wt30kfKuwWMyn3HOZnS5n7dlvIU/5BzjbUMt/3Leaizy
Lu6RV4xPZTZYP4T5PE9YX4HlPKtkPIl8OVW0N7CPfh/vvVjmnt+kteFPehFt
BkbRS2RvwZgsPhkPI1XO47zzkM2fyTsSFd5LK+0MZrq3i7wXUEifo87G1qER
063nYI15++07jQ49Z7i/iei0N5tWiKftvTO5GwxyVq86hKeaJ11fk3Gz2X93
Z1nWcmVOsm8W2u2dqT6A0WapT+bGWTkltLV4NJlRX4txQm89tG706Gemc4qx
3je8jvYiDvOHMwdgvHMfkVeOBTKz1XP6yNfD1fbNwSR799N3oolnnIyxyXvU
22zrS1DBszn5jeIm389GdTdyPKerg/G9eY/r8WtU+VbnqkuxXc9fOLMDLb7P
qbRbMYOeqscUTNfnDudvxx49jLV+PXJ4OmR3I9U3f1G9gr7Hc5p6CY54/kv2
eRSYawLPeBToebU5nsJBd/KK9ZdRqIdJyTvHjfL7eAd63iDnb/U8+vX/bvKf
gXO+pSXqanzHn5J8d5jqfhrlNWGbzC45wbNAzny1DE38bfRWXHBXuWb9z948
835Aex/f0vfR6jHNOz1l3z84oIcp7mMyGsy1NHkPSZa5GtR2NNjbzHsMmfyV
6i4s0vPP1k9jnn7qaIfxGb1H7UYn/5rk/WACfYPe7keW39FQveTjLjNebn0K
+uS8JK8R5c69jnYtlpnxI/VDFMtsV0+i1HOvOkDOSc/d5ujCDt/JJ8m3nPze
6Ifc30FsdIff0M4kM9BH8A5Hmdn/BxLF59E=
         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmWlw1eUVxv+S3JuFS27CvQSSe3P/sqOERGVzY3HpiJ0CAlrUoagsEW3H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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1mdsVlUYAOBjC21ZX6FAi7Rfq4CsFjAaGSpQXImLTaKgRhEKblGGMlQc
gAoUEBMVpCxnomG0oD8UjDLKlCVGBVyIYkVFjf5g+Nwcfjzpfe99z7nvWd/t
hSMfGvxgRgjhGerTqSSETC5pHsLTTGdn6xDW8nhWCLVsZXtRCKtZ1UijxiGc
9ffGwhBuIjsnhPXZIXzADjk7yZSzVk41P6dD+IVvc0OYx3zyxafkLZL3K8c5
Tx0lTFLDRNqzV97H9NWmkrmMLw7hFkY10z9r2S/nQDr23Y+ryFVbnXHkq6uC
0dwm53a+bBLCLJ7nUfF4OqRCONgkPvtX/B+hqTHrazXfir9Lx+u7GMlh8Tbu
cP0+71HhnU9w1AT/yEJmyXmeCd5xEe2ZZwyPskb9GXnGz+3m5A5qG5p/JvOj
+CiDkjkig0uN7RLGG9MENnFMvIXeDUKYw2zOaHeWKm0GMJAc8U/yXvT8a76i
QG2t+M3z11mSvENcQkvzMZpRZImz2eL5JB5L6O9xCqz1FvVu5UPxa8x07yBf
0MU7mxbGdaihmoVyXqbC8/xGsY+P9P82f5ij8ty49i3MU0umWY8udCXbXjnm
3jHz1sNe6c6t4t7km9MKRtNR/w2K4jg7q60TP+l/Xzrup8McYq2c6qK4Z0cx
OiGngvpqqTH+dbTRf1uq1DCAgcw2rjEMchaWsZQuydwle98c9dKuJ9vFO9Jx
7+TRnHriOu+dKG8zm7jMve7Ju7XJp4BFchYzS1096cVp8Q/cfG7dFtNVu25c
qU1d47imc9Q2lxXqGsJQquTMZJyxfcJGqu3bVyh15qcwmTX6X8BI71vDat7Q
15uc1M/ynNjvrdbjFj61Dp/RooWzL+cXKs19X8pZLV7FuuR3g2z3toprC+Pz
P9z7k1fVMZVDzk8XdZQRkvVOx/VK9kEZ/bV7gHu0GUsrctTRgCvU8av9UUcn
Y+hMmfX/xnsO8aT+X6O7vi/Limd2uHYjkt8i9e/W/nM6i0uZ5bonvTiqljO0
c92Wfd7xiNpGsN/ctzOfbTkt5xQ7Pd9Fsdx+6riqKJ7vZE+Wcpu4nEkN4zna
zLX6uo4X9Hc5V1Cjr0qCvgYygHK19WO/vp/lOSbImch6Z2AMY+krpw8Pez6O
T6gwd9eTYd4G58Q93FcdHeiohscaxpqO62s7i/LinB7nbrXdTLW66hlrJvPl
LKC+5zXeuY4Z4hVc7F43jiS1iMtp7XqbddzOcPGI4lhrK85P2idjYIh2mdQj
T3zWe0u1mcJk+rjXia1NY9sClouX8bXrbyjT9jfrvat1/K618htUwHx9LWCl
+ocyjGvEZfQwtpnMSM6gdjXM1HZGVtw3Y5O9xzDzVp8spsm5k0r76QiHmZGc
Ndro+5FUPHcrkt9hqrT5nRPcIO5aGH8f64kz+Ue7I+n4fTnNKZaKl/GZfsan
4rel2NqmS+IeK+EhauQsYYPnudqluMCcNOYmc7WYRVTKmcJgectZRjd1XFwY
v+1juYd35LzLX0n9qThn68UfpOPv2aepWNNk8ah03BNDUrHf7+2pbZzxG9K/
cfzd2iBnYzp+s0+mYr8T5Yxgk33XUm4L/jKfJ9ljzvfSRi27xX/T0XUHDrrf
ULtGRfF8J9/BuXxvrN8x1fpPyzt3FsW7WXruHA1K7our6e96Ca8ne11cx4lz
96oYrt6rec6eONAknvdaOT+w1/M9TKdUXMYO8zu1afx+rUz2aHF8/lRePBMv
Jee+KI4z+fbczWbzvrIw7ok+DeJ3vbN2zYvjfk/OTC33y7mPlvLyqeAttc1L
xzX8MxW/pc3k5HGveWpNYbL3xfm0dd0uO/4/ccJ8/k7a9U7zuYvzvDODbt53
JDf+L/c/9KJz0w==
         "],
         VertexColors->None]}]}, 
     {Hue[0.8742186457921589], EdgeForm[None], 
      GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJwV0l1syGcUx/HTdpli07dtbPoSZKU2o1xtC4kmJfOyZM3IlihtWWapVbIa
Oi7Ehek2ay8mFomko60mLryUWY1dyBZUW4ILEtlQpbTThAtki/r8L775nfye
8z/nPOf/TKhaW1aTGhHVSENvfsQ5bM2O+CAvYgUaX47oy424jbKXItbzvse0
MRFn5TajWP5F3gXUZ0QMyL2Pr+QPjI/IEO8cGZEtzkL1iIhNchvwkTqNdD+a
xH30Nq6qs9l3PyNvdMQfvEuYw29VowWP0iMO0cN4Uf0nzh8jJTNiMV2JdvMv
ogtxSFxOl+GaeB4tTe4jrnWPRhzOiljF+wap5vklmQt/ibv0GcR8vY6aqx3H
zFbouzdxWt/jvIs4wX+bzsPMURE5zqeiTk5BgRkxOSfipto3cMS9lsstR6f8
83LvY7fdVuu5Bp+572W5d7BDfgXdgH/Mv9N3P+FzfYfpMzTZ/0zfzcKfdt7N
68IL/Idv2B8m8FPVyEULv5guQImaBXQWXhHfU6Mfje6+j9eBzmQn9Hyya/P0
OD+IFDkNZj+GYfsc73wGHqlfxJuDM/bwLd2Ofjmf0HVYKV5Dv8Qu8Wz6KV4V
r1OjFlP0PcrrxmL7yeflIcuc7/HeRab8GWaZjg53HOd8LB6YYTDZgfhrcb84
RbxUfEF8D+ni13w3FrW+XUQXIt3+q+RWYlivNl43nvKX6Pkxlus77Dw7+d/u
eM75Wcy1k5P83/G+Xa11XoMD8k/wuvE6/0e5O9Cs5lSzFGGa99Dj/BZWJznJ
/8cee2hN/hsGxXVyN2KcN5BJ/1Vnr76l4kpsUadX7lMcV+cL56uxVK+7Zhnh
Td6xz2fiPlwRD3kjD5BvD2nOi1Ds3Y5R7y30qv9bUi95J2qW8OZik16j6Sj8
LecH583YbM7/6X8otJ/p9B3Mts86uhEfymmjpzAk3meWbbhhV/W87VjP/45X
j8f8iqQnevTdTTuwzR520V9RnexEbkbyzvSdRCeiSXxXvRBfN/9zOwnB/A==

         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJxFWHl4zVca/olEIrlZ7r2x3pvfZdSS2GpnLBVZqBkzlYQOo5aEocxQO6Xt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         "],
         VertexColors->None], PolygonBox[CompressedData["
1:eJwt1VeMFlUUAODZpZf9txKB3cGgCU1AdvVBRYiALiqgAlJe6BgQBAJLB0vs
AstipCQm6sKygBrpHfWBprAUsQNKZ0EsGDFBUMFvcn34kjn/nHvmnJk78zcf
PqHP+PQoisZSi4UFUTSDxg2iaDSjaC1uwIl6UdS+fhS1o30sh98aRlHnjCjq
xCfiT+mc6fdUFF2mVDyDxxy/yztsVmsL29Uew1hmigezX+2m4iZsS3L4xvEr
vMoF8UX6u25t6jCpWRSVsC47ioYxnFlyZhNbc1DNQ9zQx03aZEXRMT0e52Hx
IyzQ/wN04Zr4OrXlbZazhb7qP8MStRdTzDjxa1x0fCE7nNth3Qp+Mef91nXk
X/F5vnfcSs2W3GZdBr0dL6WcSjnz43CPfk39X0NOJxqpvUfOXlqJW/OZ42lM
Z4hZhxaEOZ/jec6odTYO/Y9iNB3EhXQ15x/u3RVK1ZrAh67xN9eJkp7V6+P8
MpayQc5GeuWYSc4NyvKj6Gn61zU3lTS1Nj8OtStYnlynqRnkTaoTRRNpzj61
PqdQvSJeZIt1S9lj9kb6zWO6eHjy7PT8eCrspWzr/vFbC7NPZQoHxYfiMGdj
mtDNDG0Kwp59ltlc1Mc+utjPb7KAbdZtj8Oaferv5y7x3Yxw3Vzy6GXdWJ4y
50jyWCanIg5rZjKLQjk5Sa45d7CdKjlb4zDTSEZwi5widju/ixLWiFezVu11
pOmvoRl+yg+9nuIkLd2DLCabfTe72CRvUUF4b7ewmRLXHMRR/bfWWytWilfF
4R19j3KGiIdyUl5ZRngnXle/kns9m3tywl4/IufjOOztucwhwzVTnNXXWywk
T7+XPfdmZhrPONarNYer6nRXr5hs67JYZk0FVxih/kjS9bVeHxtYLGdJQfhe
5FPAFDlTaSfvRzkn+FL8FaX66pQZvkcr9bKKa+7n+3yQ3Ndb7XO+1cN3tMw1
q5z5LHf+T64Qq1WDnfZxjvrZVOvjIDX99iR96SGniGHOr2MtM8Wz6K2/tFSY
Z7n5KzjlHpzODvu/o7gFOz3DHL9lU23dBX7Q/7zMMM858Re84fhoZvimHNDH
Rja5H7Uahp7GmKEndfU/kAHMtW4es/XQNhXu2QDXHMgI18slj/3WrcgPs3fz
LLpSlnwj47BPnkiFeQ7LOZQf9mc6fTiv1jleMtPLfM0lvR3hdn2VJN9rhqnV
Mw73KEqFd7s8mSt5nuK9qfANOGTdYepaszX5T2COnMlx2MfTmcYgOV2Z5f2u
qh/+Rybqox+DzbSaj7hDTluKku9jvbBf0+2BGhR6/qf1e4b51pVx05o1rCVH
nMsMz6h1VvgfGayPYubq/zjHuCo+zXrPJl1OGj3cpw5s8w7UqxueS7VaF6h2
vcUs4ZR1B+LwbdiQGWp0F98Zh//YUubxoBke4gVzdKCQmnJqsdu9mswUqlzz
AN3NedU1/6JJ8i21ttz537nMJTk/s0jeQxTTX2/3Jd8Ysw9lCGnJO0mlNf0a
hv/h/XKqeFv/j9KDBcn3i9rq1GKT6/4HRzA1Yw==
         "],
         VertexColors->None]}]}, 
     {Hue[1.], EdgeForm[None], 
      GraphicsGroupBox[{
        PolygonBox[{{7400, 7399, 3159}, {7237, 7236, 2689}, {7377, 7402, 
         3106}, {7522, 7544, 3582}, {7319, 7318, 2924}, {7632, 7652, 3976}, {
         7212, 7211, 2611}, {7184, 7185, 2546}, {7587, 7586, 3787}, {7182, 
         7181, 2532}, {7791, 7821, 4522}, {7845, 7816, 4585}, {7426, 7425, 
         3238}, {7427, 7453, 3265}, {7788, 7763, 4423}, {7292, 7291, 2846}, {
         7116, 7115, 2376}, {7497, 7496, 3473}, {7086, 7087, 2304}, {7080, 
         7079, 2298}, {7474, 7499, 3424}, {7542, 7541, 3630}, {7147, 7146, 
         2454}, {7545, 7565, 3661}, {7264, 7263, 2767}, {7118, 7119, 2385}, {
         7612, 7633, 3898}, {7473, 7472, 3395}, {7451, 7450, 3316}, {7375, 
         7374, 3081}, {7351, 7350, 3003}, {7718, 7739, 4289}, {7520, 7519, 
         3552}, {7630, 7629, 3945}, {7676, 7697, 4133}, {7610, 7609, 3866}, {
         7738, 7766, 4367}, {7653, 7677, 4055}, {7566, 7589, 3740}, {7815, 
         7789, 4504}, {7674, 7673, 4103}, {7651, 7650, 4024}, {7818, 7858, 
         4598}, {7564, 7563, 3709}, {7765, 7793, 4445}, {7500, 7521, 3503}, {
         7696, 7719, 4211}, {7590, 7611, 3819}},
         VertexColors->None], PolygonBox[CompressedData["
1:eJxNmnm81tP2x59bqHOe8zzP9+niujo9DZJEKFNuZLyaB4lmKTKkEd2EUhFF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         "],
         VertexColors->None], 
        PolygonBox[{{7472, 7451, 3316, 3396, 3395}, {7629, 7610, 3866, 3946, 
         3945}, {7764, 7765, 4445, 4365, 4366}, {7790, 7791, 4522, 4442, 
         4443}, {7650, 7630, 3945, 4025, 4024}, {7789, 7788, 4423, 4424, 
         4504}, {7609, 7587, 3787, 3867, 3866}, {7350, 7319, 2924, 3004, 
         3003}, {7737, 7738, 4367, 4287, 4288}, {7673, 7651, 4024, 4104, 
         4103}, {7376, 7351, 3003, 3083, 3082}, {7149, 7184, 2546, 2545, 
         2465}, {7402, 7403, 3185, 3105, 3106}, {7425, 7400, 3159, 3239, 
         3238}, {7499, 7500, 3503, 3423, 3424}, {7401, 7375, 3081, 3161, 
         3160}, {7652, 7653, 4055, 3975, 3976}, {7453, 7454, 3344, 3264, 
         3265}, {7631, 7632, 3976, 3896, 3897}, {7588, 7564, 3709, 3789, 
         3788}, {7589, 7590, 3819, 3739, 3740}, {7563, 7542, 3630, 3710, 
         3709}, {7611, 7612, 3898, 3818, 3819}, {7675, 7676, 4133, 4053, 
         4054}, {7695, 7696, 4211, 4131, 4132}, {7565, 7566, 3740, 3660, 
         3661}, {7265, 7237, 2689, 2769, 2768}, {7291, 7264, 2767, 2847, 
         2846}, {7320, 7292, 2846, 2926, 2925}, {7452, 7426, 3238, 3318, 
         3317}, {7148, 7116, 2376, 2456, 2455}, {7543, 7520, 3552, 3632, 
         3631}, {7238, 7212, 2611, 2691, 2690}, {7117, 7080, 2298, 2378, 
         2377}, {7183, 7147, 2454, 2534, 2533}, {7519, 7497, 3473, 3553, 
         3552}, {7498, 7473, 3395, 3475, 3474}, {7544, 7545, 3661, 3581, 
         3582}, {7816, 7815, 4504, 4505, 4585}, {7521, 7522, 3582, 3502, 
         3503}, {7817, 7818, 4598, 4518, 4519}, {7087, 7118, 2385, 2384, 
         2304}, {7211, 7182, 2532, 2612, 2611}, {7717, 7718, 4289, 4209, 
         4210}},
         VertexColors->None]}]}}, {{}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{7079, 7080, 7117, 7115, 7116, 7148, 7146, 7147, 7183, 7181, 
         7182, 7211, 7212, 7238, 7236, 7237, 7265, 7263, 7264, 7291, 7292, 
         7320, 7318, 7319, 7350, 7351, 7376, 7374, 7375, 7401, 7399, 7400, 
         7425, 7426, 7452, 7450, 7451, 7472, 7473, 7498, 7496, 7497, 7519, 
         7520, 7543, 7541, 7542, 7563, 7564, 7588, 7586, 7587, 7609, 7610, 
         7629, 7630, 7650, 7651, 7673, 7674, 7694, 7716, 7736, 7763, 7788, 
         7789, 7815, 7816, 7845, 7846, 7847, 7848, 7849, 7850, 7851, 7852, 
         7853, 7854, 7855, 7856, 7857, 7858, 7818, 7817, 7819, 7820, 7821, 
         7791, 7790, 7792, 7793, 7765, 7764, 7766, 7738, 7737, 7739, 7718, 
         7717, 7719, 7696, 7695, 7697, 7676, 7675, 7677, 7653, 7652, 7632, 
         7631, 7633, 7612, 7611, 7590, 7589, 7566, 7565, 7545, 7544, 7522, 
         7521, 7500, 7499, 7474, 7454, 7453, 7427, 7403, 7402, 7377, 7352, 
         7321, 7293, 7266, 7239, 7213, 7185, 7184, 7149, 7119, 7118, 7087, 
         7086, 7085, 7084, 7083, 7082, 7081, 7079}],
        "0.019799999999999998`"],
       Annotation[#, 0.019799999999999998`, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6894, 6895, 6932, 6931, 6929, 6930, 6961, 6959, 6960, 6987, 
         6988, 7018, 7016, 7017, 7055, 7053, 7054, 7077, 7078, 7114, 7112, 
         7113, 7144, 7145, 7180, 7178, 7179, 7209, 7210, 7235, 7233, 7234, 
         7261, 7262, 7290, 7288, 7289, 7316, 7317, 7349, 7347, 7348, 7372, 
         7373, 7398, 7396, 7397, 7424, 7422, 7423, 7449, 7447, 7448, 7470, 
         7471, 7495, 7493, 7494, 7518, 7516, 7517, 7539, 7540, 7562, 7560, 
         7561, 7584, 7585, 7607, 7608, 7627, 7628, 7648, 7649, 7671, 7672, 
         7693, 7714, 7715, 7735, 7762, 7787, 7814, 7843, 7844, 7880, 7881, 
         7907, 7908, 7909, 7934, 7935, 7936, 7937, 7967, 7968, 7969, 7970, 
         7971, 7972, 7973, 7974, 7975, 7976, 7977, 7939, 7938, 7940, 7941, 
         7942, 7943, 7911, 7910, 7912, 7913, 7914, 7883, 7882, 7884, 7885, 
         7860, 7859, 7861, 7823, 7822, 7824, 7795, 7794, 7796, 7768, 7767, 
         7769, 7741, 7740, 7742, 7721, 7720, 7722, 7699, 7698, 7679, 7678, 
         7680, 7655, 7654, 7635, 7634, 7614, 7613, 7592, 7591, 7568, 7567, 
         7547, 7546, 7524, 7523, 7501, 7476, 7475, 7455, 7429, 7428, 7404, 
         7378, 7354, 7353, 7322, 7294, 7267, 7240, 7214, 7186, 7150, 7120, 
         7089, 7088, 7056, 7020, 7019, 6989, 6963, 6962, 6935, 6934, 6933, 
         6899, 6898, 6897, 6896, 6894}],
        "0.0165`"],
       Annotation[#, 0.0165, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6751, 6752, 6781, 6779, 6780, 6809, 6807, 6808, 6836, 6834, 
         6835, 6861, 6862, 6893, 6891, 6892, 6927, 6928, 6957, 6958, 6986, 
         6984, 6985, 7014, 7015, 7052, 7050, 7051, 7075, 7076, 7110, 7111, 
         7143, 7141, 7142, 7176, 7177, 7208, 7206, 7207, 7231, 7232, 7260, 
         7258, 7259, 7286, 7287, 7315, 7313, 7314, 7346, 7344, 7345, 7371, 
         7369, 7370, 7394, 7395, 7421, 7419, 7420, 7446, 7444, 7445, 7469, 
         7467, 7468, 7492, 7490, 7491, 7514, 7515, 7538, 7536, 7537, 7558, 
         7559, 7583, 7581, 7582, 7605, 7606, 7626, 7646, 7647, 7669, 7670, 
         7692, 7712, 7713, 7734, 7761, 7786, 7813, 7841, 7842, 7879, 7905, 
         7906, 7932, 7933, 7965, 7966, 7997, 7998, 7999, 8022, 8023, 8024, 
         8025, 8056, 8057, 8058, 8059, 8060, 8061, 8062, 8063, 8064, 8065, 
         8066, 8067, 8027, 8026, 8028, 8029, 8030, 8031, 8001, 8000, 8002, 
         8003, 8004, 7979, 7978, 7980, 7981, 7945, 7944, 7946, 7947, 7916, 
         7915, 7917, 7887, 7886, 7888, 7889, 7863, 7862, 7864, 7826, 7825, 
         7827, 7798, 7797, 7771, 7770, 7772, 7744, 7743, 7745, 7724, 7723, 
         7701, 7700, 7702, 7682, 7681, 7657, 7656, 7637, 7636, 7616, 7615, 
         7594, 7593, 7570, 7569, 7548, 7526, 7525, 7502, 7478, 7477, 7456, 
         7431, 7430, 7405, 7379, 7355, 7324, 7323, 7295, 7268, 7241, 7215, 
         7187, 7151, 7121, 7090, 7057, 7022, 7021, 6990, 6964, 6936, 6901, 
         6900, 6863, 6838, 6837, 6811, 6810, 6783, 6782, 6758, 6757, 6756, 
         6755, 6754, 6753, 6751}],
        "0.0132`"],
       Annotation[#, 0.0132, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6641, 6642, 6667, 6665, 6666, 6689, 6687, 6688, 6708, 6706, 
         6707, 6730, 6731, 6749, 6750, 6778, 6776, 6777, 6805, 6806, 6832, 
         6833, 6859, 6860, 6889, 6890, 6926, 6924, 6925, 6955, 6956, 6982, 
         6983, 7012, 7013, 7049, 7047, 7048, 7073, 7074, 7109, 7107, 7108, 
         7139, 7140, 7174, 7175, 7205, 7203, 7204, 7230, 7228, 7229, 7256, 
         7257, 7285, 7283, 7284, 7312, 7310, 7311, 7343, 7341, 7342, 7368, 
         7366, 7367, 7393, 7391, 7392, 7418, 7416, 7417, 7443, 7441, 7442, 
         7466, 7464, 7465, 7488, 7489, 7513, 7511, 7512, 7534, 7535, 7556, 
         7557, 7579, 7580, 7603, 7604, 7624, 7625, 7645, 7667, 7668, 7691, 
         7711, 7733, 7760, 7785, 7812, 7840, 7878, 7904, 7930, 7931, 7964, 
         7995, 7996, 8020, 8021, 8053, 8054, 8055, 8092, 8093, 8094, 8116, 
         8117, 8118, 8119, 8120, 8121, 8147, 8148, 8149, 8150, 8151, 8152, 
         8153, 8154, 8155, 8123, 8122, 8124, 8125, 8126, 8127, 8128, 8096, 
         8095, 8097, 8098, 8099, 8069, 8068, 8070, 8071, 8072, 8033, 8032, 
         8034, 8035, 8006, 8005, 8007, 7983, 7982, 7984, 7949, 7948, 7950, 
         7951, 7919, 7918, 7920, 7891, 7890, 7892, 7866, 7865, 7829, 7828, 
         7830, 7800, 7799, 7801, 7774, 7773, 7747, 7746, 7748, 7726, 7725, 
         7704, 7703, 7684, 7683, 7659, 7658, 7639, 7638, 7618, 7617, 7596, 
         7595, 7572, 7571, 7549, 7528, 7527, 7503, 7480, 7479, 7457, 7432, 
         7406, 7381, 7380, 7356, 7325, 7296, 7269, 7242, 7216, 7188, 7152, 
         7122, 7091, 7058, 7023, 6991, 6965, 6937, 6903, 6902, 6864, 6839, 
         6812, 6785, 6784, 6759, 6732, 6710, 6709, 6691, 6690, 6669, 6668, 
         6649, 6648, 6647, 6646, 6645, 6644, 6643, 6641}],
        "0.009899999999999999`"],
       Annotation[#, 0.009899999999999999, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1E1MzwEYB/AfihS9mUOlJMa4KIzJcBHFTDI5sLEQNu8zNsymv1Bxytsy
sil5N3GwKVNpxoYwb1svOOCAXsjbycccPs/xOXyflxEFm/I29gmCYIgSSgiC
fZyjmrNUcZs6nvCIx7TRThdf6WRAYhBEkEwK6WSQTQ5LWcZ2dlDOEa5yjQc8
5BMf+EhkUhBEMZZxzCKLAlayn30UU8kZGrlLA+94T/iwIOhHGGmkMoJMprGE
xeSzlc1s4TBlHOIyF7lEM/dopIl2WmnjJ738IDY5CKKJYTRjmMl0ZrCYfNaz
gSJCVHCdWu7TSjc9RKbIl0ksYC0hKmmmg98kDg+CJDLJYxHbqOAktdyghac8
4wtf6aSLvqlyI4xw+jOACAaSxkhGEUUkgxhMNDHEEkc8Q+jRt5tvfKeXH7TR
SjsdNNJAE/eopopz1FBKCWWsZhWFrCGHbOYygQwmEkcs8fTIqJuXvOAVddym
nhMcZxc72c0KljObLOaQzvh/eZNAf8L5YlafecNrmrjKFY5xlD2sYyG5TCWN
gXTZn05eUs9p9lLIPDIYSrf9fs5NytlMLpNJ5Jc7esotThGiiLXMZwqjiCGa
P275LR20cId6LnCeGsoopYSDHGA/xQn//8hfcSGpig==
         "]],
        "0.0066`"],
       Annotation[#, 0.0066, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[LineBox[CompressedData["
1:eJwl1FtMz2EYB/BfNmc2m8pSSiPZnOrfGRcIoQP9y/mckRjNCDdOcYFhYzbF
ZFyQMFy4YFPMcWOM2ajM2ZWcbWpufMzF571+n+/7fN/Essrw2oggCIY5hscE
wQjCFDOdGZSznBVsooqN7KCak5ziKtd4wENaeUkbn+ncPwi6EEsco5hMHnOZ
xwaqOMghznOBe9znAx/pHhsEXelGMkOZQC5LWcwSqtnJcepo4jqNvOEVr+kU
JwQiiCeBHLLIppQSiglTyRrWspfd7OEs9ZzmDDdp4gattPCCZn7xg5/0GhAE
vRlMEmMYSwmlrGYHNdRyibu08o3u8UEwgDQKKaKcauq4xW1e0U5MgvkoZj01
1HKZRzzmE20EA+VCAgNJJIVUQoxjPBPIZSKTmMwsZjOHucxjPgtYyCIWM4U8
pjKNfAooJJ00Msgki2yGkEQyQ4kikmj60e7uv+ngLW94x3vucod7nKWeBvaz
jwNUsJJVFJBPIRmkk0kkfYnil1x/0swLWmiikaPUcoytbKGMpSxjKlMIkUoc
sXSjK1+98RdauMNtLnKE7WxjFWFGM4ie9OC7/XpOIyeoppwCQkT/20MdeMYV
DrOOYrKIpUPnnnKVOnZRQRE5ZJNEH/7o+lue0MQ5GtjHZipYyUxKKSGbLDLJ
IJ00QqSSwihGxvz/p/4CGGepmA==
         "]],
        "0.0033`"],
       Annotation[#, 0.0033, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6401, 6402, 6434, 6442, 6443, 6478, 6499, 6500, 6516, 6527, 
         6528, 6560, 6583, 6584, 6602, 6617, 6618, 6634, 6635, 6660, 6681, 
         6682, 6701, 6702, 6724, 6725, 6744, 6770, 6771, 6799, 6800, 6826, 
         6827, 6853, 6854, 6883, 6881, 6882, 6917, 6918, 6949, 6950, 6976, 
         6977, 7006, 7004, 7005, 7040, 7041, 7068, 7066, 7067, 7100, 7098, 
         7099, 7133, 7131, 7132, 7167, 7165, 7166, 7197, 7195, 7196, 7222, 
         7221, 7219, 7220, 7248, 7247, 7245, 7246, 7276, 7275, 7274, 7272, 
         7273, 7303, 7302, 7301, 7299, 7300, 7333, 7332, 7331, 7330, 7328, 
         7329}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{7756, 7754, 7755, 7782, 7780, 7781, 7809, 7807, 7808, 7836, 
         7837, 7874, 7872, 7873, 7901, 7899, 7900, 7927, 7925, 7926, 7960, 
         7958, 7959, 7991, 7992, 8016, 8014, 8015, 8045, 8043, 8044, 8081, 
         8079, 8080, 8107, 8108, 8138, 8136, 8137, 8166, 8164, 8165, 8188, 
         8186, 8187, 8218, 8216, 8217, 8235, 8233, 8234, 8251, 8249, 8250, 
         8272, 8270, 8271, 8290, 8288, 8289, 8305, 8303, 8304, 8313, 8311, 
         8312, 8321, 8320, 8318, 8319, 8330, 8329, 8327, 8328, 8340, 8339, 
         8337, 8338}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{8302, 8301, 8308, 8309, 8310, 8314, 8315, 8316, 8317, 8324, 
         8325, 8326, 8333, 8334, 8335, 8336}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ], 
      TagBox[
       TooltipBox[LineBox[{8322, 8323, 8331, 8332, 8341, 8342}],
        "0"],
       Annotation[#, 0, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6403, 6404, 6437, 6435, 6436, 6457, 6458, 6486, 6506, 6507, 
         6521, 6541, 6572, 6592, 6610, 6626, 6652, 6673, 6694, 6714, 6735, 
         6762, 6789, 6815, 6842, 6867, 6906, 6940, 6968, 6969, 6994, 7026, 
         7061, 7062, 7094, 7125, 7126, 7155, 7156, 7157, 7158, 7191, 7192, 
         7193, 7194, 7160, 7159, 7161, 7162, 7163, 7164, 7128, 7127, 7129, 
         7130, 7096, 7095, 7097, 7064, 7063, 7065, 7038, 7037, 7039, 7002, 
         7001, 7003, 6975, 6974, 6947, 6946, 6948, 6916, 6915, 6880, 6879, 
         6852, 6851, 6825, 6824, 6798, 6797, 6769, 6768, 6743, 6742, 6723, 
         6722, 6700, 6680, 6679, 6659, 6658, 6633, 6616, 6601, 6600, 6582, 
         6559, 6526, 6515, 6514, 6498, 6497, 6477, 6476, 6475, 6441, 6440, 
         6439, 6438, 6433, 6432, 6431, 6430, 6429, 6428, 6427, 6426, 6425, 
         6424, 6423, 6422, 6421, 6420, 6419, 6418, 6417, 6416, 6415, 6414, 
         6413, 6412, 6411, 6410, 6409, 6408, 6407, 6406, 6405, 6403}],
        RowBox[{"-", "0.0033`"}]],
       Annotation[#, -0.0033, "Tooltip"]& ], 
      TagBox[
       TooltipBox[
        LineBox[{8046, 8047, 8085, 8084, 8082, 8083, 8111, 8109, 8110, 8139, 
         8140, 8169, 8167, 8168, 8189, 8190, 8219, 8220, 8236, 8237, 8252, 
         8253, 8273, 8274, 8291, 8292, 8293, 8294, 8306, 8307, 8296, 8295, 
         8297, 8298, 8299, 8300, 8276, 8275, 8277, 8255, 8254, 8239, 8238, 
         8222, 8221, 8191, 8170, 8141, 8112, 8087, 8086, 8050, 8049, 8048, 
         8046}],
        RowBox[{"-", "0.0033`"}]],
       Annotation[#, -0.0033, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6459, 6460, 6490, 6489, 6487, 6488, 6510, 6508, 6509, 6522, 
         6523, 6542, 6573, 6574, 6593, 6611, 6627, 6653, 6674, 6695, 6696, 
         6715, 6716, 6736, 6763, 6790, 6816, 6843, 6868, 6869, 6907, 6941, 
         6942, 6970, 6995, 6996, 6997, 7027, 7028, 7029, 7030, 7031, 7032, 
         7033, 7034, 7035, 7036, 6999, 6998, 7000, 6972, 6971, 6973, 6944, 
         6943, 6945, 6913, 6912, 6914, 6878, 6877, 6850, 6849, 6822, 6821, 
         6823, 6796, 6795, 6767, 6741, 6740, 6721, 6720, 6699, 6678, 6677, 
         6657, 6632, 6631, 6615, 6599, 6598, 6581, 6558, 6557, 6525, 6524, 
         6513, 6512, 6511, 6496, 6495, 6494, 6493, 6492, 6491, 6474, 6473, 
         6472, 6471, 6470, 6469, 6468, 6467, 6466, 6465, 6464, 6463, 6462, 
         6461, 6459}],
        RowBox[{"-", "0.0066`"}]],
       Annotation[#, -0.0066, "Tooltip"]& ]}, 
     {GrayLevel[0], Opacity[0.4], 
      TagBox[
       TooltipBox[
        LineBox[{6543, 6544, 6577, 6575, 6576, 6594, 6595, 6612, 6613, 6628, 
         6654, 6675, 6697, 6717, 6737, 6764, 6791, 6792, 6817, 6844, 6845, 
         6870, 6871, 6872, 6908, 6909, 6910, 6911, 6874, 6873, 6875, 6876, 
         6847, 6846, 6848, 6819, 6818, 6820, 6794, 6793, 6766, 6765, 6739, 
         6738, 6719, 6718, 6698, 6676, 6656, 6655, 6630, 6629, 6614, 6597, 
         6596, 6580, 6579, 6578, 6556, 6555, 6554, 6553, 6552, 6551, 6550, 
         6549, 6548, 6547, 6546, 6545, 6543}],
        RowBox[{"-", "0.009899999999999999`"}]],
       Annotation[#, -0.009899999999999999, "Tooltip"]& ]}, {}, {}}}],
  AspectRatio->1,
  Frame->True,
  PlotRange->{{1., 80.}, {1., 80.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 183},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->410111908]
}, Open  ]],

Cell[TextData[{
 "It should be noted that this solution is only available at discrete points \
in the interior of the region. It does not include the value on the boundary \
or at other points in the interior. If the solution was to be used for other \
computational purposes this might be a limitation. ",
 Cell[BoxData[
  ButtonBox["PadLeft",
   BaseStyle->"Link",
   ButtonData->
    "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#782554218"]], 
  "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PadRight",
   BaseStyle->"Link",
   ButtonData->
    "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#782554218"]], 
  "InlineFormula"],
 " could be used to add the boundary value and ",
 Cell[BoxData[
  ButtonBox["Interpolation",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Interpolation"]], "InlineFormula"],
 " used to generate an interpolating function for the region."
}], "Text",
 CellID->705583400],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sol", "=", 
   RowBox[{"PadRight", "[", 
    RowBox[{
     RowBox[{"PadLeft", "[", 
      RowBox[{"sol", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"num", "+", "1"}], ",", 
         RowBox[{"num", "+", "1"}]}], "}"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"num", "+", "2"}], ",", 
       RowBox[{"num", "+", "2"}]}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"Flatten", "[", 
    RowBox[{
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"i", " ", "hStep"}], ",", " ", 
         RowBox[{"j", " ", "hStep"}], ",", " ", 
         RowBox[{"sol", "[", 
          RowBox[{"[", 
           RowBox[{
            RowBox[{"i", "+", "1"}], ",", 
            RowBox[{"j", "+", "1"}]}], "]"}], "]"}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"j", ",", " ", "0", ",", " ", 
         RowBox[{"num", "+", "1"}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"i", ",", "0", ",", 
         RowBox[{"num", "+", "1"}]}], "}"}]}], "]"}], ",", " ", "1"}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{"iFun", "=", 
  RowBox[{"Interpolation", "[", "data", "]"}]}]}], "Input",
 CellLabel->"In[15]:=",
 CellID->335505066],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0.`", ",", "1.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "1.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
   "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->335257658]
}, Open  ]],

Cell[TextData[{
 "This uses ",
 Cell[BoxData[
  ButtonBox["Plot3D",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Plot3D"]], "InlineFormula"],
 " to plot the solution."
}], "Text",
 CellID->628068116],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"iFun", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->1058700607],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 133},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->5905303]
}, Open  ]],

Cell["\<\
Now the solution can be computed at any position in the region.\
\>", "Text",
 CellID->323427151],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"iFun", "[", 
    RowBox[{"x", ",", "0.5"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1", ",", "0.05"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->559016300],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.9005301754018364`*^-20"}], ",", "0.0014137413923502645`", 
   ",", "0.003014080952288059`", ",", "0.004819591501628617`", ",", 
   "0.006838930883026156`", ",", "0.009060376746933961`", ",", 
   "0.011447581843680487`", ",", "0.013938211604168136`", ",", 
   "0.016443986071033655`", ",", "0.018851753145758703`", ",", 
   "0.021025831170119427`", ",", "0.02281215517588819`", ",", 
   "0.024044842932597794`", ",", "0.024555701760089137`", ",", 
   "0.02418691914309858`", ",", "0.022806687331003744`", ",", 
   "0.02032675899676287`", ",", "0.01671986918448203`", ",", 
   "0.012033545850943319`", ",", "0.0063950443154589184`", ",", "0.`"}], 
  "}"}]], "Output",
 ImageSize->{559, 54},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->82396169]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Mesh Partitioning", "Section",
 CellTags->{"t:215", "c:161"},
 CellID->1370044451],

Cell["\<\
This example demonstrates a practical use for extreme eigenvalues. The aim is \
to partition the vertices of a mesh/graph into two subdomains, with each part \
having roughly equal numbers of vertices and with the partition cutting \
through as few edges as possible. (An edge is cut if its two vertices lie in \
each of the two subdomains.) One way to partition a graph is to form the \
Laplacian and use its Fiedler vector (the eigenvector that corresponds to the \
second smallest eigenvalue) to order the vertices. There are more efficient \
algorithms for graph partitioning that do not require the calculation of the \
eigenvector, but the approach using Fiedler remains an important one.\
\>", "Text",
 CellID->1017424823],

Cell[CellGroupData[{

Cell["Data", "Subsubsection",
 CellTags->{"t:216", "c:162"},
 CellID->1594202747],

Cell["This is the data for the coordinates of the vertices", "Text",
 CellID->1759588857],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"coordinates", " ", "=", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.809475", ",", "12.5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "12.809475"}], ",", 
        RowBox[{"-", "12.5"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7."}], ",", 
        RowBox[{"-", "4."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7."}], ",", "2."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", "0."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.", ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", "0."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.", ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1."}], ",", "7.415715"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.", ",", 
        RowBox[{"-", "7.415715"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7."}], ",", 
        RowBox[{"-", "1."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.", ",", "8.716814"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1."}], ",", 
        RowBox[{"-", "13.474961"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4.769961", ",", "25.682898"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.", ",", "13.474961"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.769961"}], ",", 
        RowBox[{"-", "25.682898"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.736733", ",", 
        RowBox[{"-", "23.013988"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "8.206816"}], ",", 
        RowBox[{"-", "29.954422"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.", ",", "1.037611"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.", ",", "13.301991"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8.822488", ",", "14.823734"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7.494958"}], ",", 
        RowBox[{"-", "19.97955"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "2.746936"}], ",", 
        RowBox[{"-", "18.232192"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"13.830102", ",", "22.165298"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7.494958", ",", "19.97955"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.746936", ",", "18.232192"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "13.830102"}], ",", 
        RowBox[{"-", "22.165298"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "17.681008"}], ",", 
        RowBox[{"-", "21.469559"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7.905112"}], ",", 
        RowBox[{"-", "27.965817"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "22.988602"}], ",", 
        RowBox[{"-", "13.396162"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8.897511", ",", 
        RowBox[{"-", "15.808074"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6.245633", ",", "15.852277"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "11.891051"}], ",", 
        RowBox[{"-", "17.475286"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"11.891052", ",", "17.475286"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "17.238035"}], ",", 
        RowBox[{"-", "9.740396"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1."}], ",", 
        RowBox[{"-", "3.029623"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1."}], ",", "17.861054"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.", ",", "3.029623"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1."}], ",", "2.193046"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.", ",", 
        RowBox[{"-", "2.193046"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.", ",", 
        RowBox[{"-", "17.861054"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.", ",", 
        RowBox[{"-", "12.638385"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1."}], ",", 
        RowBox[{"-", "8.252292"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.", ",", "8.252292"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.561635"}], ",", 
        RowBox[{"-", "22.748417"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "9.895431"}], ",", 
        RowBox[{"-", "25.457646"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.937437"}], ",", 
        RowBox[{"-", "26.879819"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.", ",", 
        RowBox[{"-", "11.226769"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4.923133", ",", 
        RowBox[{"-", "17.356056"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.197251", ",", 
        RowBox[{"-", "26.32948"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.647938", ",", 
        RowBox[{"-", "25.24711"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "2.153142"}], ",", 
        RowBox[{"-", "29.248212"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "18.935854"}], ",", 
        RowBox[{"-", "24.655299"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.", ",", "7.169801"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "14.955767"}], ",", 
        RowBox[{"-", "17.159894"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "18.991451"}], ",", 
        RowBox[{"-", "15.547121"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "13.641527"}], ",", 
        RowBox[{"-", "25.994504"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.95189"}], ",", 
        RowBox[{"-", "23.991475"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "10.89993"}], ",", 
        RowBox[{"-", "27.348592"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.851524"}], ",", 
        RowBox[{"-", "27.806102"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "18.098166"}], ",", 
        RowBox[{"-", "2.47478"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", 
        RowBox[{"-", "30."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.049083"}], ",", "28.76261"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.467927"}], ",", 
        RowBox[{"-", "12.32765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.049083"}], ",", 
        RowBox[{"-", "1.23739"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "30."}], ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "30.5"}], ",", "38.930527"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "9.467927"}], ",", 
        RowBox[{"-", "42.32765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "9.549083"}], ",", "7.693137"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.103408"}], ",", 
        RowBox[{"-", "44.977211"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4."}], ",", "9.930527"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.598625", ",", 
        RowBox[{"-", "43.16474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.5"}], ",", "38.930527"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6.915051", ",", "41.082649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.853493", ",", 
        RowBox[{"-", "1.654037"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"31.598625", ",", 
        RowBox[{"-", "43.16474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"34.448756", ",", 
        RowBox[{"-", "37.904037"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"30.", ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"34.448756", ",", 
        RowBox[{"-", "7.904037"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"36.915051", ",", "11.082649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", "30."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "20.024542"}], ",", "23.311832"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"21.915051", ",", "41.082649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"36.915051", ",", "41.082649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", 
        RowBox[{"-", "15."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", "30."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.467927"}], ",", 
        RowBox[{"-", "42.32765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.524542"}], ",", "14.381305"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.524542"}], ",", "44.381305"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "41.494301"}], ",", 
        RowBox[{"-", "6.698081"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.467927"}], ",", 
        RowBox[{"-", "27.32765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "28.573625"}], ",", "13.143915"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "30."}], ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "29.536812"}], ",", "26.037221"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "29.20189"}], ",", 
        RowBox[{"-", "18.491474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "28.573625"}], ",", 
        RowBox[{"-", "1.856085"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "24.467927"}], ",", 
        RowBox[{"-", "42.32765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.75"}], ",", "28.39579"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "19.061354"}], ",", "10.418526"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15."}], ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "14.20189"}], ",", 
        RowBox[{"-", "33.491474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.25"}], ",", "49.465263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "10.512271"}], ",", "20.586443"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "6.155112"}], ",", 
        RowBox[{"-", "37.465817"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "3.128275"}], ",", 
        RowBox[{"-", "37.112712"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.733963"}], ",", 
        RowBox[{"-", "51.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.076571"}], ",", 
        RowBox[{"-", "44.624106"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.397938", ",", 
        RowBox[{"-", "34.74711"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.75"}], ",", "28.39579"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.384981", ",", "42.841449"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.372577", ",", "31.623973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.", ",", 
        RowBox[{"-", "3.547565"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.957526", ",", "50.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"15.799313", ",", 
        RowBox[{"-", "51.58237"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"17.397938", ",", 
        RowBox[{"-", "34.74711"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"17.872577", ",", "31.623973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"21.673134", ",", 
        RowBox[{"-", "26.856056"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"15.", ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"21.673134", ",", 
        RowBox[{"-", "11.856056"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"23.629115", ",", "2.297981"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"25.372577", ",", "16.623973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"18.823003", ",", 
        RowBox[{"-", "32.116759"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"25.372577", ",", "31.623973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"25.957526", ",", "50.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"30.", ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"36.404738", ",", "6.25"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"36.915051", ",", "26.082649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"47.224378", ",", 
        RowBox[{"-", "48.952019"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"45.", ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"47.224378", ",", 
        RowBox[{"-", "18.952019"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", 
        RowBox[{"-", "30."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", "45."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", "15."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", "15."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", "45."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "60."}], ",", 
        RowBox[{"-", "45."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.733963"}], ",", 
        RowBox[{"-", "21.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.733963"}], ",", 
        RowBox[{"-", "6.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.524542"}], ",", "29.381305"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.524542"}], ",", 
        RowBox[{"-", "0.618695"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "45.25"}], ",", "49.465263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.733963"}], ",", 
        RowBox[{"-", "36.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "45."}], ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.049083"}], ",", "13.76261"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "40.012271"}], ",", "41.655916"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "32.241451"}], ",", 
        RowBox[{"-", "10.047122"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "31.018842"}], ",", 
        RowBox[{"-", "7.316776"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "29.20189"}], ",", 
        RowBox[{"-", "33.491474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "33.811354"}], ",", "20.953263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "33.811354"}], ",", "5.953263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "34.733963"}], ",", 
        RowBox[{"-", "51.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "31.228264"}], ",", 
        RowBox[{"-", "12.861906"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "29.20189"}], ",", 
        RowBox[{"-", "25.991474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "25.262271"}], ",", "31.121179"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "30.25"}], ",", "49.465263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "23.335896"}], ",", "5.334568"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "19.733963"}], ",", 
        RowBox[{"-", "51.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "21.70189"}], ",", 
        RowBox[{"-", "33.491474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "14.786812"}], ",", "15.502485"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.5"}], ",", "38.930527"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15."}], ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "12.233963"}], ",", 
        RowBox[{"-", "51.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "8.375"}], ",", "23.128422"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "8.125"}], ",", "33.663159"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.25"}], ",", "49.465263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "2.051704"}], ",", 
        RowBox[{"-", "52.488606"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.614857"}], ",", 
        RowBox[{"-", "36.936159"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.799313", ",", 
        RowBox[{"-", "51.58237"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.577471", ",", "34.262174"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.457526", ",", "50.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7.899656", ",", 
        RowBox[{"-", "55.791185"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.498282", ",", 
        RowBox[{"-", "38.955925"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.842506", ",", "33.382774"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.297594", ",", 
        RowBox[{"-", "30.538295"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.435192", ",", 
        RowBox[{"-", "26.592768"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"11.010127", ",", 
        RowBox[{"-", "29.223119"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"15.", ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"18.457526", ",", "50.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.598625", ",", 
        RowBox[{"-", "43.16474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"15.285322", ",", 
        RowBox[{"-", "21.332065"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"22.651124", ",", 
        RowBox[{"-", "4.779037"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"25.210814", ",", 
        RowBox[{"-", "37.640749"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"24.607106", ",", "9.375"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"28.060945", ",", 
        RowBox[{"-", "32.380046"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"24.862263", ",", "11.791324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"25.372577", ",", "24.123973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"23.698969", ",", 
        RowBox[{"-", "47.373555"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"28.060945", ",", 
        RowBox[{"-", "17.380046"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"34.448756", ",", 
        RowBox[{"-", "22.904037"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"33.457526", ",", "50.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"40.957526", ",", "50.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"40.836567", ",", 
        RowBox[{"-", "43.428028"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"45.799313", ",", 
        RowBox[{"-", "51.58237"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"47.224378", ",", 
        RowBox[{"-", "3.952019"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.457526", ",", "5.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.457526", ",", "20.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.457526", ",", "35.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"47.224378", ",", 
        RowBox[{"-", "33.952019"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.457526", ",", "50.541324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", 
        RowBox[{"-", "15."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"60.", ",", 
        RowBox[{"-", "45."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "54.762271"}], ",", "37.190653"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "54.762271"}], ",", "52.190653"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "54.762271"}], ",", "7.190653"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "54.762271"}], ",", "22.190653"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.733963"}], ",", 
        RowBox[{"-", "51.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "50.74715"}], ",", 
        RowBox[{"-", "3.349041"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.733963"}], ",", 
        RowBox[{"-", "43.663825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.733963"}], ",", 
        RowBox[{"-", "28.663825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "49.733963"}], ",", 
        RowBox[{"-", "13.663825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "44.286812"}], ",", "6.571958"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "44.286812"}], ",", "36.571958"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "44.286812"}], ",", "21.571958"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "45."}], ",", "60."}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "42.233963"}], ",", 
        RowBox[{"-", "51.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "37.875"}], ",", "44.197895"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.049083"}], ",", "6.26261"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "37.625"}], ",", "54.732632"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "31.967927"}], ",", 
        RowBox[{"-", "42.32765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "27.233963"}], ",", 
        RowBox[{"-", "51.163825"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "22.875"}], ",", "44.197895"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "22.625"}], ",", "54.732632"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "23.125"}], ",", "33.663159"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7.875"}], ",", "44.197895"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7.625"}], ",", "54.732632"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.538286"}], ",", 
        RowBox[{"-", "52.312053"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.19249", ",", "51.420725"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8.698969", ",", 
        RowBox[{"-", "47.373555"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"22.899656", ",", 
        RowBox[{"-", "55.791185"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"24.498282", ",", 
        RowBox[{"-", "38.955925"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"26.635879", ",", 
        RowBox[{"-", "35.010398"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"30.799313", ",", 
        RowBox[{"-", "51.58237"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"37.899656", ",", 
        RowBox[{"-", "55.791185"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"40.836567", ",", 
        RowBox[{"-", "28.428028"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"40.836567", ",", 
        RowBox[{"-", "13.428028"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"39.411502", ",", 
        RowBox[{"-", "46.058379"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"45.", ",", 
        RowBox[{"-", "60."}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.202369", ",", "3.125"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.457526", ",", "43.041324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.457526", ",", "13.041324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"48.457526", ",", "28.041324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"53.612189", ",", 
        RowBox[{"-", "54.476009"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"53.612189", ",", 
        RowBox[{"-", "9.476009"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"53.612189", ",", 
        RowBox[{"-", "39.476009"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"53.612189", ",", 
        RowBox[{"-", "24.476009"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "52.625"}], ",", "54.732632"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"38.698969", ",", 
        RowBox[{"-", "47.373555"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"52.899656", ",", 
        RowBox[{"-", "55.791185"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.467927"}], ",", 
        RowBox[{"-", "34.82765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "39.467927"}], ",", 
        RowBox[{"-", "19.82765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "33.811354"}], ",", 
        RowBox[{"-", "1.546737"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "34.334909"}], ",", 
        RowBox[{"-", "30.409562"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "28.573625"}], ",", "5.643915"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "27.108433"}], ",", 
        RowBox[{"-", "13.129034"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "25.781113"}], ",", 
        RowBox[{"-", "7.626123"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "24.068872"}], ",", 
        RowBox[{"-", "21.573387"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "24.068872"}], ",", 
        RowBox[{"-", "29.073387"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "26.834909"}], ",", 
        RowBox[{"-", "37.909562"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "23.335896"}], ",", 
        RowBox[{"-", "2.165432"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "18.57976"}], ",", "3.971873"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "19.334909"}], ",", 
        RowBox[{"-", "37.909562"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "16.568872"}], ",", 
        RowBox[{"-", "29.073387"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "18.716608"}], ",", 
        RowBox[{"-", "18.592506"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.900463"}], ",", 
        RowBox[{"-", "14.023561"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "16.967927"}], ",", 
        RowBox[{"-", "42.32765"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.755555"}], ",", 
        RowBox[{"-", "21.817429"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "13.823625"}], ",", "2.609179"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "14.805062"}], ",", 
        RowBox[{"-", "19.755279"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "13.735814"}], ",", 
        RowBox[{"-", "24.079901"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "14.085381"}], ",", 
        RowBox[{"-", "28.34564"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "12.133847"}], ",", 
        RowBox[{"-", "24.135439"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "12.868763"}], ",", 
        RowBox[{"-", "5.533022"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "8.837372"}], ",", 
        RowBox[{"-", "36.141036"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "8.25"}], ",", "28.39579"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "8."}], ",", "38.930527"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7.351243"}], ",", 
        RowBox[{"-", "25.992286"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "5.274542"}], ",", "12.777095"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "6.904738"}], ",", 
        RowBox[{"-", "10.376146"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "6.904738"}], ",", 
        RowBox[{"-", "12.987481"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4."}], ",", "0.596523"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4."}], ",", 
        RowBox[{"-", "2.014812"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "5.667546"}], ",", 
        RowBox[{"-", "33.533567"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.875"}], ",", "23.128422"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.625"}], ",", "33.663159"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4."}], ",", 
        RowBox[{"-", "6.126146"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "2.424183"}], ",", 
        RowBox[{"-", "24.562157"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1.249536"}], ",", 
        RowBox[{"-", "24.814118"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.277258"}], ",", 
        RowBox[{"-", "36.206476"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.087549", ",", 
        RowBox[{"-", "22.881202"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.868367", ",", 
        RowBox[{"-", "20.437521"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.797594", ",", 
        RowBox[{"-", "30.538295"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.134981", ",", "32.306713"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.981226", ",", "38.551811"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.961567", ",", 
        RowBox[{"-", "14.99722"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4.193324", ",", 
        RowBox[{"-", "20.325447"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.873468", ",", "15.767091"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.548454", ",", 
        RowBox[{"-", "41.060333"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6.347766", ",", 
        RowBox[{"-", "32.642703"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4.927766", ",", "19.630794"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7.571269", ",", "28.653436"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8.158723", ",", "17.401642"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7.068019", ",", 
        RowBox[{"-", "21.621736"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7.816221", ",", 
        RowBox[{"-", "26.461124"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.875502", ",", 
        RowBox[{"-", "8.731056"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.101339", ",", "26.894636"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.897938", ",", 
        RowBox[{"-", "34.74711"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.", ",", "11.782909"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"11.831484", ",", "5.422981"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.900312", ",", "15.089181"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.393814", ",", "36.353311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.666352", ",", 
        RowBox[{"-", "21.200421"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"14.415051", ",", "41.082649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.263313", ",", 
        RowBox[{"-", "6.755046"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"15.851339", ",", "26.894636"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"15.285322", ",", 
        RowBox[{"-", "13.832065"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"18.219295", ",", "7.398991"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"19.091026", ",", "14.561987"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"19.601339", ",", "26.894636"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"17.241304", ",", "0.321972"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"21.673134", ",", 
        RowBox[{"-", "19.356056"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"18.708291", ",", "10.9375"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"19.601339", ",", "19.394636"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"19.893814", ",", "36.353311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"29.415051", ",", "41.082649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"30.016926", ",", "4.273991"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"31.143814", ",", "13.853311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"31.143814", ",", "28.853311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"31.143814", ",", "21.353311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"29.038936", ",", 
        RowBox[{"-", "2.803028"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"36.915051", ",", "18.582649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"36.915051", ",", "33.582649"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "27.615026"}], ",", 
        RowBox[{"-", "11.721642"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.098625", ",", 
        RowBox[{"-", "43.16474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "27.003722"}], ",", 
        RowBox[{"-", "10.356469"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "34.334909"}], ",", 
        RowBox[{"-", "22.909562"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "34.334909"}], ",", 
        RowBox[{"-", "15.409562"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "29.20189"}], ",", 
        RowBox[{"-", "22.241474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "23.528737"}], ",", 
        RowBox[{"-", "17.484774"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "25.95476"}], ",", "1.739241"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "24.068872"}], ",", 
        RowBox[{"-", "25.323387"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "24.299083"}], ",", "18.227874"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "21.93964"}], ",", 
        RowBox[{"-", "5.050452"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "21.602822"}], ",", 
        RowBox[{"-", "7.46107"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "22.054429"}], ",", 
        RowBox[{"-", "19.418871"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "20.717031"}], ",", "1.429894"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "20.318872"}], ",", 
        RowBox[{"-", "29.073387"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "17.95189"}], ",", 
        RowBox[{"-", "33.491474"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "18.48769"}], ",", 
        RowBox[{"-", "12.531142"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "16.318387"}], ",", 
        RowBox[{"-", "19.314727"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.960896"}], ",", "0.067199"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.023755"}], ",", 
        RowBox[{"-", "11.120198"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "12.839119"}], ",", 
        RowBox[{"-", "9.016511"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "14.305219"}], ",", "9.055832"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "11.834909"}], ",", 
        RowBox[{"-", "37.909562"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "15.625"}], ",", "33.663159"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "12.983163"}], ",", "0.839197"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "11.204353"}], ",", 
        RowBox[{"-", "31.722948"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "11.768479"}], ",", 
        RowBox[{"-", "25.726075"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "9.904737"}], ",", 
        RowBox[{"-", "8.25"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "10.152217"}], ",", 
        RowBox[{"-", "16.239775"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "8.900272"}], ",", 
        RowBox[{"-", "26.711731"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "9.857106"}], ",", 
        RowBox[{"-", "11.438073"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "10.030677"}], ",", "14.13979"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "7.778206"}], ",", 
        RowBox[{"-", "15.366096"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "5.5"}], ",", "5.965263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "6.952369"}], ",", 
        RowBox[{"-", "7.188073"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.5625"}], ",", "25.762106"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.4375"}], ",", "31.029475"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "3.952369"}], ",", 
        RowBox[{"-", "11.925554"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4."}], ",", 
        RowBox[{"-", "3.514812"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "3.952369"}], ",", 
        RowBox[{"-", "9.314219"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "3.353699"}], ",", 
        RowBox[{"-", "26.281359"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.825837"}], ",", 
        RowBox[{"-", "15.609836"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.6875"}], ",", "20.494738"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4.3125"}], ",", "36.296843"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "4."}], ",", "4.707857"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "0.375"}], ",", "44.197895"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1."}], ",", 
        RowBox[{"-", "5.640958"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.620987", ",", 
        RowBox[{"-", "20.433219"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.561635", ",", "22.748417"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.947481", ",", "24.40566"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.452471", ",", "28.994805"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.596662", ",", 
        RowBox[{"-", "19.093251"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1.359079", ",", "16.044508"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"2.961567", ",", 
        RowBox[{"-", "17.608555"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.980783", ",", 
        RowBox[{"-", "13.111995"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.", ",", "13.388476"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4.173716", ",", "29.972536"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"4.496285", ",", "17.042234"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.", ",", 
        RowBox[{"-", "14.292864"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.171805", ",", "14.810988"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.358865", ",", 
        RowBox[{"-", "24.161617"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"5.306234", ",", "29.532836"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6.870296", ",", "17.915913"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6.747423", ",", 
        RowBox[{"-", "28.433888"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8.936288", ",", "45.811987"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8.643814", ",", "36.353311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7.741415", ",", "15.8575"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6.706887", ",", "31.018105"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8.43565", ",", "26.288767"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.751605", ",", 
        RowBox[{"-", "24.107252"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.969867", ",", "19.213434"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.35677", ",", "16.14951"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"11.236958", ",", "29.259304"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.815981", ",", "13.661867"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.895432", ",", "25.457646"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.904737", ",", "10.608407"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.926747", ",", "3.531389"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"10.8614", ",", "14.956458"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.32048", ",", "8.961491"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.580412", ",", 
        RowBox[{"-", "11.28156"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"13.111958", ",", "29.259304"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"12.091417", ",", 
        RowBox[{"-", "18.57007"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"14.122577", ",", "31.623973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"14.536394", ",", "2.872477"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"14.840721", ",", "24.529967"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"15.514385", ",", "9.949495"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.143814", ",", "36.353311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"14.955767", ",", "17.159894"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.436288", ",", "45.811987"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.715721", ",", "24.529967"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"17.054163", ",", 
        RowBox[{"-", "26.724412"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.752309", ",", 
        RowBox[{"-", "3.216537"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.460564", ",", "18.363642"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"18.968223", ",", 
        RowBox[{"-", "9.305551"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"16.861958", ",", "29.259304"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"18.835869", ",", "12.145662"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"19.601339", ",", "23.144636"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"20.924205", ",", "4.848486"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"21.413201", ",", "8.386995"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"21.622577", ",", "31.623973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"22.231801", ",", "15.59298"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"25.372577", ",", "27.873973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"25.372577", ",", "20.373973"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"27.393814", ",", "36.353311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"23.936288", ",", "45.811987"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"27.312016", ",", "6.824495"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"28.060945", ",", 
        RowBox[{"-", "24.880046"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"30.505922", ",", "7.8125"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"31.143814", ",", "36.353311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"31.143814", ",", "17.603311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"31.143814", ",", "25.103311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"30.761079", ",", "10.228824"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"41.814558", ",", "1.148991"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "5.756135"}], ",", "19.223748"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"3.", ",", "10.777142"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"7.103689", ",", 
        RowBox[{"-", "27.7763"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"9.", ",", "2.584625"}], "}"}]}], "}"}]}], ";"}], 
  "\n"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->412616762],

Cell["This is a list of triangles that forms the mesh.", "Text",
 CellID->625222771],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"triangles", "=", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"209", ",", "90", ",", "138"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"206", ",", "139", ",", "90"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"207", ",", "1", ",", "139"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"208", ",", "138", ",", "9"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"142", ",", "9", ",", "89"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"141", ",", "89", ",", "66"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"210", ",", "140", ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"146", ",", "66", ",", "140"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"250", ",", "1", ",", "207"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"206", ",", "93", ",", "139"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"207", ",", "139", ",", "93"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"208", ",", "92", ",", "138"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"209", ",", "138", ",", "92"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"208", ",", "9", ",", "144"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"206", ",", "90", ",", "143"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"209", ",", "143", ",", "90"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"211", ",", "9", ",", "142"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"211", ",", "144", ",", "9"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"212", ",", "146", ",", "140"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"213", ",", "141", ",", "66"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"213", ",", "66", ",", "146"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"214", ",", "142", ",", "89"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"214", ",", "89", ",", "141"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"212", ",", "140", ",", "210"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"250", ",", "218", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"210", ",", "2", ",", "147"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"250", ",", "207", ",", "145"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"211", ",", "142", ",", "144"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"207", ",", "93", ",", "145"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"215", ",", "92", ",", "208"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"144", ",", "215", ",", "208"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"216", ",", "93", ",", "206"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"143", ",", "216", ",", "206"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"209", ",", "92", ",", "217"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"143", ",", "209", ",", "217"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"250", ",", "145", ",", "218"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"210", ",", "219", ",", "212"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"142", ",", "214", ",", "94"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"142", ",", "94", ",", "144"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"146", ",", "253", ",", "213"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"141", ",", "254", ",", "214"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"212", ",", "91", ",", "146"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"213", ",", "95", ",", "141"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"219", ",", "210", ",", "147"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"149", ",", "145", ",", "93"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"216", ",", "149", ",", "93"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"215", ",", "144", ",", "69"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"216", ",", "143", ",", "67"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"217", ",", "67", ",", "143"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"215", ",", "148", ",", "92"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"217", ",", "92", ",", "148"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"219", ",", "91", ",", "212"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"214", ",", "68", ",", "94"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"144", ",", "94", ",", "69"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"253", ",", "95", ",", "213"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"254", ",", "68", ",", "214"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"254", ",", "141", ",", "95"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"253", ",", "146", ",", "91"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"222", ",", "218", ",", "145"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"216", ",", "67", ",", "149"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"220", ",", "145", ",", "149"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"221", ",", "215", ",", "69"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"221", ",", "148", ",", "215"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"219", ",", "147", ",", "155"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"145", ",", "220", ",", "222"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"153", ",", "67", ",", "217"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"148", ",", "153", ",", "217"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"219", ",", "155", ",", "91"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"255", ",", "69", ",", "94"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"256", ",", "95", ",", "253"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"339", ",", "254", ",", "95"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"340", ",", "68", ",", "254"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"150", ",", "94", ",", "68"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"222", ",", "97", ",", "218"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"221", ",", "154", ",", "148"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"221", ",", "69", ",", "154"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"253", ",", "91", ",", "223"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"155", ",", "147", ",", "70"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"149", ",", "67", ",", "71"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"220", ",", "149", ",", "71"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"254", ",", "339", ",", "340"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"339", ",", "95", ",", "256"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"255", ",", "154", ",", "69"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"151", ",", "255", ",", "94"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"223", ",", "91", ",", "155"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"150", ",", "68", ",", "340"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"220", ",", "159", ",", "222"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"150", ",", "151", ",", "94"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"256", ",", "253", ",", "152"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"153", ",", "98", ",", "67"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"154", ",", "96", ",", "148"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"153", ",", "148", ",", "96"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"152", ",", "253", ",", "223"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"98", ",", "71", ",", "67"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"220", ",", "71", ",", "159"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"222", ",", "159", ",", "97"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"339", ",", "99", ",", "340"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"157", ",", "339", ",", "256"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"156", ",", "150", ",", "340"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"154", ",", "255", ",", "257"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"340", ",", "99", ",", "156"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"155", ",", "224", ",", "223"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"255", ",", "151", ",", "100"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"256", ",", "152", ",", "157"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"341", ",", "339", ",", "157"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"341", ",", "99", ",", "339"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"224", ",", "155", ",", "70"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"336", ",", "150", ",", "156"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"257", ",", "96", ",", "154"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"255", ",", "100", ",", "257"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"338", ",", "151", ",", "150"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"262", ",", "152", ",", "223"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"345", ",", "98", ",", "153"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"258", ",", "156", ",", "99"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"336", ",", "338", ",", "150"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"96", ",", "345", ",", "153"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"258", ",", "336", ",", "156"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"259", ",", "100", ",", "151"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"158", ",", "71", ",", "98"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"338", ",", "259", ",", "151"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"224", ",", "101", ",", "223"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"225", ",", "159", ",", "71"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"262", ",", "223", ",", "101"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"343", ",", "257", ",", "100"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"226", ",", "97", ",", "159"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"157", ",", "344", ",", "341"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"261", ",", "157", ",", "152"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"341", ",", "260", ",", "99"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"338", ",", "336", ",", "258"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"257", ",", "160", ",", "96"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"258", ",", "99", ",", "342"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"227", ",", "71", ",", "158"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"343", ",", "100", ",", "263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"343", ",", "160", ",", "257"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"262", ",", "162", ",", "152"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"263", ",", "100", ",", "259"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"344", ",", "260", ",", "341"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"344", ",", "157", ",", "261"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"34", ",", "338", ",", "258"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"224", ",", "70", ",", "161"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"342", ",", "99", ",", "260"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"225", ",", "71", ",", "227"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"338", ",", "34", ",", "259"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"159", ",", "225", ",", "226"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"261", ",", "152", ",", "162"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"158", ",", "98", ",", "86"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"345", ",", "86", ",", "98"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"34", ",", "258", ",", "342"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"345", ",", "96", ",", "103"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"224", ",", "161", ",", "101"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"346", ",", "263", ",", "259"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"160", ",", "103", ",", "96"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"262", ",", "101", ",", "265"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"347", ",", "259", ",", "34"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"343", ",", "263", ",", "349"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"160", ",", "343", ",", "349"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"342", ",", "260", ",", "348"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"346", ",", "259", ",", "347"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"342", ",", "348", ",", "34"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"261", ",", "350", ",", "344"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"162", ",", "262", ",", "265"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"226", ",", "165", ",", "97"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"344", ",", "57", ",", "260"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"350", ",", "261", ",", "162"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"348", ",", "260", ",", "57"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"161", ",", "70", ",", "104"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"227", ",", "158", ",", "102"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"346", ",", "65", ",", "263"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"350", ",", "57", ",", "344"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"349", ",", "264", ",", "160"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"349", ",", "263", ",", "65"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"346", ",", "347", ",", "65"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"164", ",", "225", ",", "227"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"269", ",", "101", ",", "161"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"158", ",", "86", ",", "102"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"264", ",", "103", ",", "160"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"265", ",", "101", ",", "269"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"225", ",", "106", ",", "226"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"162", ",", "351", ",", "350"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"86", ",", "345", ",", "163"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"351", ",", "162", ",", "265"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"345", ",", "103", ",", "163"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"350", ",", "266", ",", "57"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"264", ",", "349", ",", "354"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"351", ",", "266", ",", "350"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"349", ",", "65", ",", "354"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"359", ",", "227", ",", "102"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"359", ",", "164", ",", "227"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"225", ",", "164", ",", "106"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"352", ",", "60", ",", "268"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"226", ",", "106", ",", "165"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"353", ",", "267", ",", "32"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"267", ",", "59", ",", "60"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"357", ",", "103", ",", "264"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"351", ",", "265", ",", "105"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"355", ",", "39", ",", "352"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"353", ",", "59", ",", "267"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"268", ",", "60", ",", "59"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"353", ",", "32", ",", "270"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"266", ",", "274", ",", "57"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"268", ",", "355", ",", "352"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"270", ",", "32", ",", "62"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"161", ",", "166", ",", "269"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"351", ",", "105", ",", "266"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"354", ",", "271", ",", "264"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"357", ",", "163", ",", "103"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"269", ",", "358", ",", "265"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"354", ",", "65", ",", "360"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"166", ",", "161", ",", "104"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"272", ",", "353", ",", "270"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"271", ",", "357", ",", "264"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"107", ",", "102", ",", "86"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"353", ",", "272", ",", "59"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"270", ",", "62", ",", "31"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"358", ",", "105", ",", "265"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"163", ",", "107", ",", "86"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"355", ",", "356", ",", "39"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"266", ",", "105", ",", "274"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"270", ",", "31", ",", "272"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"355", ",", "268", ",", "6"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"268", ",", "59", ",", "6"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"273", ",", "31", ",", "62"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"273", ",", "62", ",", "61"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"356", ",", "276", ",", "39"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"354", ",", "360", ",", "271"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"355", ",", "6", ",", "356"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"273", ",", "275", ",", "31"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"359", ",", "102", ",", "278"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"273", ",", "61", ",", "275"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"361", ",", "274", ",", "105"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"359", ",", "168", ",", "164"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"163", ",", "357", ",", "367"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"166", ",", "72", ",", "269"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"228", ",", "106", ",", "164"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"358", ",", "269", ",", "72"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"229", ",", "165", ",", "106"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"357", ",", "271", ",", "73"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"362", ",", "275", ",", "61"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"362", ",", "61", ",", "63"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"356", ",", "363", ",", "276"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6", ",", "366", ",", "356"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"367", ",", "107", ",", "163"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"358", ",", "277", ",", "105"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"364", ",", "6", ",", "37"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"167", ",", "102", ",", "107"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"361", ",", "105", ",", "277"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"357", ",", "73", ",", "367"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"360", ",", "8", ",", "271"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"362", ",", "50", ",", "275"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"361", ",", "22", ",", "274"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"6", ",", "364", ",", "366"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"366", ",", "363", ",", "356"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"362", ",", "63", ",", "50"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"278", ",", "102", ",", "167"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"168", ",", "359", ",", "278"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"166", ",", "104", ",", "110"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"279", ",", "164", ",", "168"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"279", ",", "228", ",", "164"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"229", ",", "106", ",", "228"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"271", ",", "8", ",", "73"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"358", ",", "72", ",", "277"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"363", ",", "7", ",", "276"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"365", ",", "50", ",", "63"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"364", ",", "37", ",", "26"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"364", ",", "368", ",", "366"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"365", ",", "63", ",", "33"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"366", ",", "282", ",", "363"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"449", ",", "107", ",", "367"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"365", ",", "280", ",", "50"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"72", ",", "166", ",", "74"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"277", ",", "286", ",", "361"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"364", ",", "26", ",", "368"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"286", ",", "22", ",", "361"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"367", ",", "73", ",", "281"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"449", ",", "167", ",", "107"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"283", ",", "366", ",", "368"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"277", ",", "72", ",", "108"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"365", ",", "33", ",", "280"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"370", ",", "7", ",", "363"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"370", ",", "363", ",", "282"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"366", ",", "283", ",", "282"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"229", ",", "10", ",", "165"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"369", ",", "73", ",", "8"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"371", ",", "278", ",", "167"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"166", ",", "110", ",", "74"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"281", ",", "449", ",", "367"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"372", ",", "168", ",", "278"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"277", ",", "108", ",", "286"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"379", ",", "279", ",", "168"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"280", ",", "33", ",", "64"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"377", ",", "368", ",", "26"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"110", ",", "104", ",", "12"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"108", ",", "72", ",", "74"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"377", ",", "283", ",", "368"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"369", ",", "75", ",", "73"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"281", ",", "73", ",", "75"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"378", ",", "167", ",", "449"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"285", ",", "15", ",", "7"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"284", ",", "8", ",", "15"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"370", ",", "289", ",", "7"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"282", ",", "375", ",", "370"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"373", ",", "282", ",", "283"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"167", ",", "378", ",", "371"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"372", ",", "278", ",", "371"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"20", ",", "280", ",", "64"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"168", ",", "372", ",", "379"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"380", ",", "369", ",", "8"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"279", ",", "77", ",", "228"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"286", ",", "56", ",", "22"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"169", ",", "229", ",", "228"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"283", ",", "377", ",", "373"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"27", ",", "377", ",", "26"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"8", ",", "284", ",", "380"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"285", ",", "284", ",", "15"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"374", ",", "7", ",", "289"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"374", ",", "285", ",", "7"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"286", ",", "108", ",", "109"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"375", ",", "289", ",", "370"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"375", ",", "282", ",", "373"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"108", ",", "74", ",", "109"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"376", ",", "20", ",", "64"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"379", ",", "77", ",", "279"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"449", ",", "281", ",", "41"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"378", ",", "449", ",", "41"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"376", ",", "290", ",", "20"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"369", ",", "13", ",", "75"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"380", ",", "13", ",", "369"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"286", ",", "109", ",", "171"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"281", ",", "75", ",", "41"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"376", ",", "64", ",", "51"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"378", ",", "287", ",", "371"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"111", ",", "74", ",", "110"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"377", ",", "17", ",", "373"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"113", ",", "372", ",", "371"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"372", ",", "288", ",", "379"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"289", ",", "382", ",", "374"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"284", ",", "43", ",", "380"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"285", ",", "40", ",", "284"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"374", ",", "40", ",", "285"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"375", ",", "47", ",", "289"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"375", ",", "373", ",", "47"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"381", ",", "228", ",", "77"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"377", ",", "27", ",", "17"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"381", ",", "169", ",", "228"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"109", ",", "74", ",", "111"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"292", ",", "56", ",", "286"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"170", ",", "111", ",", "110"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"376", ",", "51", ",", "290"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"171", ",", "292", ",", "286"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"170", ",", "110", ",", "12"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"169", ",", "231", ",", "229"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"378", ",", "41", ",", "287"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"231", ",", "10", ",", "229"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"371", ",", "287", ",", "113"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"382", ",", "40", ",", "374"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"284", ",", "40", ",", "43"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"380", ",", "43", ",", "13"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"382", ",", "289", ",", "47"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"75", ",", "13", ",", "41"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"373", ",", "17", ",", "47"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"171", ",", "109", ",", "111"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"372", ",", "113", ",", "288"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"291", ",", "290", ",", "51"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"379", ",", "288", ",", "77"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"291", ",", "49", ",", "290"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"230", ",", "111", ",", "170"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"230", ",", "170", ",", "12"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"291", ",", "51", ",", "55"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"384", ",", "287", ",", "41"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"291", ",", "55", ",", "49"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"171", ",", "111", ",", "76"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"230", ",", "172", ",", "111"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"292", ",", "171", ",", "76"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"292", ",", "112", ",", "56"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"230", ",", "12", ",", "172"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"385", ",", "113", ",", "287"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"288", ",", "113", ",", "296"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"383", ",", "49", ",", "293"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"293", ",", "49", ",", "55"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"172", ",", "76", ",", "111"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"381", ",", "77", ",", "114"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"385", ",", "287", ",", "384"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"297", ",", "77", ",", "288"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"174", ",", "169", ",", "381"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"295", ",", "56", ",", "112"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"294", ",", "383", ",", "293"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"295", ",", "54", ",", "56"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"387", ",", "45", ",", "383"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"174", ",", "231", ",", "169"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"293", ",", "55", ",", "21"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"113", ",", "385", ",", "386"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"231", ",", "174", ",", "10"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"386", ",", "296", ",", "113"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"297", ",", "114", ",", "77"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"298", ",", "46", ",", "45"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"450", ",", "19", ",", "48"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"294", ",", "387", ",", "383"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"296", ",", "173", ",", "288"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"388", ",", "19", ",", "391"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"114", ",", "174", ",", "381"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"293", ",", "21", ",", "294"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"173", ",", "297", ",", "288"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"389", ",", "45", ",", "387"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"292", ",", "76", ",", "301"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"389", ",", "298", ",", "45"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"44", ",", "23", ",", "42"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"44", ",", "14", ",", "23"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"46", ",", "52", ",", "14"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"42", ",", "58", ",", "48"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"391", ",", "19", ",", "450"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"385", ",", "384", ",", "18"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"112", ",", "292", ",", "301"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"390", ",", "46", ",", "298"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"388", ",", "300", ",", "30"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"300", ",", "388", ",", "391"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"387", ",", "294", ",", "299"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"172", ",", "12", ",", "175"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"294", ",", "21", ",", "299"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"450", ",", "48", ",", "58"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"386", ",", "392", ",", "296"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"390", ",", "52", ",", "46"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"385", ",", "18", ",", "386"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"389", ",", "387", ",", "53"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"389", ",", "53", ",", "298"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"52", ",", "23", ",", "14"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"42", ",", "23", ",", "58"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"391", ",", "450", ",", "24"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"232", ",", "76", ",", "172"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"393", ",", "30", ",", "300"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"295", ",", "112", ",", "302"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"173", ",", "296", ",", "177"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"387", ",", "299", ",", "53"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"390", ",", "298", ",", "394"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"300", ",", "391", ",", "395"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"296", ",", "392", ",", "177"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"393", ",", "303", ",", "30"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"297", ",", "78", ",", "114"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"399", ",", "54", ",", "295"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"394", ",", "298", ",", "53"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"397", ",", "392", ",", "386"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"450", ",", "58", ",", "24"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"391", ",", "24", ",", "395"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"18", ",", "397", ",", "386"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"300", ",", "395", ",", "393"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"390", ",", "394", ",", "52"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"174", ",", "117", ",", "10"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"174", ",", "114", ",", "400"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"401", ",", "297", ",", "173"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"396", ",", "54", ",", "451"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"76", ",", "232", ",", "301"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"302", ",", "399", ",", "295"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"395", ",", "36", ",", "393"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"392", ",", "397", ",", "403"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"303", ",", "393", ",", "398"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"177", ",", "392", ",", "403"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"399", ",", "451", ",", "54"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"175", ",", "232", ",", "172"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"301", ",", "176", ",", "112"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"393", ",", "36", ",", "398"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"397", ",", "18", ",", "304"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"177", ",", "401", ",", "173"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"114", ",", "78", ",", "400"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"112", ",", "176", ",", "302"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"401", ",", "78", ",", "297"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"398", ",", "29", ",", "303"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"403", ",", "397", ",", "304"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"307", ",", "396", ",", "451"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"404", ",", "304", ",", "18"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"398", ",", "36", ",", "402"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"306", ",", "396", ",", "405"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"398", ",", "305", ",", "29"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"305", ",", "398", ",", "402"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"175", ",", "12", ",", "181"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"403", ",", "115", ",", "177"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"396", ",", "307", ",", "405"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"404", ",", "18", ",", "410"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"232", ",", "337", ",", "301"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"400", ",", "117", ",", "174"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"399", ",", "302", ",", "178"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"301", ",", "337", ",", "176"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"399", ",", "178", ",", "451"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"403", ",", "304", ",", "115"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"401", ",", "177", ",", "115"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"406", ",", "29", ",", "305"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"176", ",", "310", ",", "302"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"451", ",", "180", ",", "307"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"117", ",", "122", ",", "10"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"305", ",", "402", ",", "407"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"302", ",", "310", ",", "178"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"315", ",", "35", ",", "306"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"402", ",", "25", ",", "407"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"304", ",", "404", ",", "408"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"405", ",", "315", ",", "306"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"308", ",", "116", ",", "35"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"412", ",", "16", ",", "452"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"401", ",", "314", ",", "78"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"451", ",", "178", ",", "180"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"305", ",", "407", ",", "406"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"409", ",", "25", ",", "311"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"452", ",", "116", ",", "79"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"411", ",", "311", ",", "16"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"408", ",", "115", ",", "304"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"408", ",", "404", ",", "410"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"412", ",", "452", ",", "79"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"405", ",", "307", ",", "179"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"407", ",", "25", ",", "413"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"400", ",", "78", ",", "316"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"413", ",", "25", ",", "409"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"311", ",", "411", ",", "409"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"412", ",", "312", ",", "16"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"307", ",", "180", ",", "179"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"415", ",", "308", ",", "35"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"401", ",", "115", ",", "314"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"417", ",", "35", ",", "315"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"407", ",", "38", ",", "406"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"411", ",", "16", ",", "414"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"232", ",", "175", ",", "118"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"405", ",", "179", ",", "315"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"407", ",", "413", ",", "38"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"414", ",", "16", ",", "312"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"309", ",", "408", ",", "410"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"314", ",", "316", ",", "78"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"337", ",", "232", ",", "183"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"411", ",", "5", ",", "409"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"413", ",", "409", ",", "313"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"408", ",", "416", ",", "115"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"308", ",", "317", ",", "116"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"337", ",", "183", ",", "176"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"412", ",", "79", ",", "419"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"413", ",", "313", ",", "38"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"411", ",", "414", ",", "5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"317", ",", "79", ",", "116"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"417", ",", "319", ",", "35"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"312", ",", "412", ",", "419"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"117", ",", "400", ",", "424"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"408", ",", "309", ",", "416"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"409", ",", "5", ",", "313"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"415", ",", "35", ",", "319"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"310", ",", "176", ",", "119"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"309", ",", "410", ",", "420"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"418", ",", "314", ",", "115"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"310", ",", "119", ",", "178"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"416", ",", "418", ",", "115"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"417", ",", "315", ",", "184"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"315", ",", "179", ",", "184"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"410", ",", "28", ",", "420"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"175", ",", "181", ",", "118"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"178", ",", "119", ",", "180"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"415", ",", "317", ",", "308"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"414", ",", "312", ",", "421"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"400", ",", "316", ",", "424"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"426", ",", "179", ",", "180"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"414", ",", "421", ",", "5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"416", ",", "309", ",", "318"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"232", ",", "118", ",", "183"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"419", ",", "79", ",", "323"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"422", ",", "314", ",", "418"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"417", ",", "184", ",", "319"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"420", ",", "318", ",", "309"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"422", ",", "316", ",", "314"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"176", ",", "183", ",", "119"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"427", ",", "79", ",", "317"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"430", ",", "418", ",", "416"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"182", ",", "122", ",", "117"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"419", ",", "320", ",", "312"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"426", ",", "184", ",", "179"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"427", ",", "323", ",", "79"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"428", ",", "28", ",", "423"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"420", ",", "28", ",", "425"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"421", ",", "312", ",", "320"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"318", ",", "430", ",", "416"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"424", ",", "182", ",", "117"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"415", ",", "319", ",", "429"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"423", ",", "5", ",", "321"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"126", ",", "426", ",", "180"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"28", ",", "428", ",", "425"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"431", ",", "5", ",", "421"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"180", ",", "119", ",", "126"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"420", ",", "425", ",", "318"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"317", ",", "415", ",", "429"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"120", ",", "422", ",", "418"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"430", ",", "120", ",", "418"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"422", ",", "327", ",", "316"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"428", ",", "423", ",", "321"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"431", ",", "321", ",", "5"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"425", ",", "322", ",", "318"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"319", ",", "184", ",", "324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"430", ",", "318", ",", "322"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"421", ",", "320", ",", "325"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"419", ",", "323", ",", "433"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"424", ",", "316", ",", "87"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"428", ",", "432", ",", "425"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"325", ",", "431", ",", "421"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"320", ",", "419", ",", "433"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"233", ",", "118", ",", "181"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"422", ",", "120", ",", "327"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"324", ",", "184", ",", "426"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"430", ",", "322", ",", "120"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"428", ",", "321", ",", "326"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"326", ",", "432", ",", "428"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"427", ",", "317", ",", "185"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"425", ",", "432", ",", "322"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"429", ",", "319", ",", "123"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"191", ",", "183", ",", "118"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"327", ",", "87", ",", "316"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"427", ",", "185", ",", "323"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"426", ",", "126", ",", "121"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"429", ",", "185", ",", "317"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"434", ",", "325", ",", "320"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"234", ",", "119", ",", "183"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"123", ",", "319", ",", "324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"182", ",", "424", ",", "440"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"431", ",", "325", ",", "434"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"435", ",", "120", ",", "322"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"435", ",", "327", ",", "120"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"182", ",", "128", ",", "122"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"436", ",", "321", ",", "431"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"121", ",", "324", ",", "426"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"434", ",", "320", ",", "433"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"436", ",", "326", ",", "321"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"186", ",", "126", ",", "119"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"433", ",", "323", ",", "124"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"424", ",", "87", ",", "440"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"233", ",", "191", ",", "118"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"429", ",", "123", ",", "185"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"323", ",", "185", ",", "124"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"432", ",", "326", ",", "438"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"432", ",", "190", ",", "322"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"183", ",", "191", ",", "234"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"189", ",", "431", ",", "434"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"189", ",", "436", ",", "431"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"435", ",", "322", ",", "437"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"435", ",", "127", ",", "327"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"434", ",", "433", ",", "187"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"234", ",", "186", ",", "119"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"438", ",", "326", ",", "436"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"233", ",", "181", ",", "129"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"440", ",", "128", ",", "182"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"188", ",", "121", ",", "126"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"439", ",", "87", ",", "327"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"190", ",", "432", ",", "438"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"437", ",", "322", ",", "190"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"235", ",", "126", ",", "186"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"187", ",", "189", ",", "434"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"128", ",", "82", ",", "122"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"121", ",", "442", ",", "324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"324", ",", "192", ",", "123"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"124", ",", "441", ",", "433"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"127", ",", "435", ",", "437"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"436", ",", "189", ",", "125"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"127", ",", "439", ",", "327"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"441", ",", "187", ",", "433"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"125", ",", "438", ",", "436"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"185", ",", "123", ",", "333"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"235", ",", "188", ",", "126"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"440", ",", "87", ",", "328"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"333", ",", "124", ",", "185"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"236", ",", "191", ",", "233"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"442", ",", "192", ",", "324"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"442", ",", "121", ",", "188"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"439", ",", "328", ",", "87"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"440", ",", "328", ",", "128"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"191", ",", "80", ",", "234"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"447", ",", "189", ",", "187"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"441", ",", "124", ",", "329"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"234", ",", "80", ",", "186"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"330", ",", "125", ",", "189"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"190", ",", "446", ",", "437"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"125", ",", "445", ",", "438"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"438", ",", "332", ",", "190"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"437", ",", "331", ",", "127"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"187", ",", "441", ",", "447"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"329", ",", "124", ",", "333"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"129", ",", "236", ",", "233"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"439", ",", "127", ",", "444"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"192", ",", "83", ",", "123"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"123", ",", "83", ",", "333"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"236", ",", "80", ",", "191"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"235", ",", "186", ",", "81"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"189", ",", "447", ",", "330"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"446", ",", "331", ",", "437"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"127", ",", "331", ",", "444"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"445", ",", "332", ",", "438"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"445", ",", "125", ",", "330"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"446", ",", "190", ",", "332"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"441", ",", "329", ",", "443"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"439", ",", "444", ",", "328"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"441", ",", "443", ",", "447"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"128", ",", "328", ",", "194"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"235", ",", "81", ",", "188"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"194", ",", "82", ",", "128"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"442", ",", "188", ",", "193"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"442", ",", "193", ",", "192"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"186", ",", "80", ",", "81"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"329", ",", "333", ",", "130"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"443", ",", "329", ",", "130"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"444", ",", "88", ",", "328"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"443", ",", "130", ",", "447"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"236", ",", "129", ",", "237"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"447", ",", "84", ",", "330"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"331", ",", "335", ",", "444"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"445", ",", "334", ",", "332"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"446", ",", "131", ",", "331"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"445", ",", "330", ",", "334"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"446", ",", "332", ",", "131"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"328", ",", "88", ",", "194"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"251", ",", "80", ",", "236"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"239", ",", "83", ",", "192"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"193", ",", "239", ",", "192"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"188", ",", "81", ",", "238"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"193", ",", "188", ",", "238"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"447", ",", "130", ",", "84"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"194", ",", "195", ",", "82"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"331", ",", "131", ",", "335"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"332", ",", "334", ",", "131"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"330", ",", "84", ",", "334"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"444", ",", "335", ",", "88"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"333", ",", "83", ",", "448"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"196", ",", "81", ",", "80"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"130", ",", "333", ",", "448"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"237", ",", "251", ",", "236"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"251", ",", "240", ",", "80"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"194", ",", "88", ",", "195"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"240", ",", "196", ",", "80"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"237", ",", "129", ",", "241"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"195", ",", "133", ",", "82"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"199", ",", "84", ",", "130"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"334", ",", "84", ",", "244"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"131", ",", "245", ",", "335"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"335", ",", "201", ",", "88"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"334", ",", "200", ",", "131"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"251", ",", "237", ",", "197"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"239", ",", "193", ",", "134"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"238", ",", "81", ",", "202"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"196", ",", "202", ",", "81"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"238", ",", "134", ",", "193"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"239", ",", "198", ",", "83"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"448", ",", "83", ",", "198"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"251", ",", "197", ",", "240"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"195", ",", "88", ",", "243"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"130", ",", "448", ",", "199"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"240", ",", "132", ",", "196"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"237", ",", "241", ",", "197"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"240", ",", "197", ",", "132"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"200", ",", "334", ",", "244"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"245", ",", "201", ",", "335"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"243", ",", "88", ",", "201"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"244", ",", "84", ",", "199"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"245", ",", "131", ",", "200"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"195", ",", "203", ",", "133"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"448", ",", "198", ",", "242"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"203", ",", "195", ",", "243"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"448", ",", "242", ",", "199"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"238", ",", "202", ",", "249"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"134", ",", "238", ",", "249"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"247", ",", "198", ",", "239"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"134", ",", "247", ",", "239"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"248", ",", "202", ",", "196"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"132", ",", "248", ",", "196"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"252", ",", "197", ",", "241"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"246", ",", "132", ",", "197"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"252", ",", "246", ",", "197"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"203", ",", "3", ",", "133"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"242", ",", "198", ",", "11"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"242", ",", "11", ",", "199"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"243", ",", "201", ",", "136"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"243", ",", "136", ",", "203"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"244", ",", "199", ",", "137"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"244", ",", "137", ",", "200"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"245", ",", "85", ",", "201"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"245", ",", "200", ",", "85"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"252", ",", "241", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"247", ",", "11", ",", "198"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"248", ",", "135", ",", "202"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"246", ",", "205", ",", "132"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"249", ",", "204", ",", "134"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"249", ",", "202", ",", "135"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"248", ",", "132", ",", "205"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"247", ",", "134", ",", "204"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"252", ",", "4", ",", "246"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"201", ",", "85", ",", "136"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"203", ",", "136", ",", "3"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"199", ",", "11", ",", "137"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"200", ",", "137", ",", "85"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"248", ",", "205", ",", "135"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"246", ",", "4", ",", "205"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"247", ",", "204", ",", "11"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"249", ",", "135", ",", "204"}], "}"}]}], "}"}]}], ";"}], 
  "\n"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->481627426]
}, Open  ]],

Cell[CellGroupData[{

Cell["Plotting the Mesh", "Subsection",
 CellTags->{"t:217", "c:163"},
 CellID->846283239],

Cell["To plot the mesh, simply plot each of the triangles.", "Text",
 CellID->698203792],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"edges", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"FaceToEdges", "[", 
    RowBox[{"{", 
     RowBox[{"j_", ",", "k_", ",", "l_"}], "}"}], "]"}], " ", ":=", 
   RowBox[{"{", " ", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"j", ",", "k"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "j"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"j", ",", "l"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"l", ",", "j"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "l"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"l", ",", "k"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"edges", " ", "=", " ", 
   RowBox[{"Partition", "[", 
    RowBox[{
     RowBox[{"Flatten", "[", 
      RowBox[{"Map", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"FaceToEdges", "[", "#", "]"}], ")"}], "&"}], ",", 
        "triangles"}], "]"}], "]"}], ",", 
     RowBox[{"{", "2", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"graphicsEdge", "[", 
    RowBox[{"{", 
     RowBox[{"i_", ",", "j_"}], "}"}], "]"}], ":=", " ", 
   RowBox[{"Line", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"coordinates", "[", 
       RowBox[{"[", "i", "]"}], "]"}], ",", 
      RowBox[{"coordinates", "[", 
       RowBox[{"[", "j", "]"}], "]"}]}], "}"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Lines", " ", "=", " ", 
   RowBox[{"Map", "[", 
    RowBox[{"graphicsEdge", ",", "edges"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Graphics", "[", "Lines", "]"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->860484936],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->106155769]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Laplacian", "Subsection",
 CellTags->{"t:218", "c:164"},
 CellID->455167669],

Cell[TextData[{
 "A Laplacian of a graph, ",
 Cell[BoxData[
  FormBox["L", TraditionalForm]]],
 ", is an ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "\[Cross]", "n"}], TraditionalForm]]],
 " sparse matrix, with ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]]],
 " the number of vertices in the graph. The entries of ",
 Cell[BoxData[
  FormBox["L", TraditionalForm]]],
 " are mostly zero, except that the diagonals are positive integers that \
correspond to the degree of the vertex, where the degree is the number of \
vertices linked to the vertex. In addition, if two vertices, ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox["j", TraditionalForm]]],
 ", are joined by an edge, the entry ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["L", 
    RowBox[{"i", "\[VeryThinSpace]", "j"}]], TraditionalForm]]],
 " is -1. "
}], "Text",
 CellID->666262269],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["L", 
    RowBox[{"i", "\[VeryThinSpace]", "j"}]], " ", "\[LongEqual]", " ", 
   GridBox[{
     {
      RowBox[{"degree", "(", "i", ")"}], 
      RowBox[{
       RowBox[{"if", " ", "i"}], " ", "\[LongEqual]", " ", "j"}]},
     {
      RowBox[{"-", "1"}], 
      RowBox[{
      "if", " ", "i", " ", "and", " ", "j", " ", "form", " ", "an", " ", 
       "edge"}]},
     {"0", "otherwise"}
    }]}], TraditionalForm]], "NumberedEquation",
 GridBoxOptions->{
 GridBoxAlignment->{
  "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
   "RowsIndexed" -> {}}},
 CellID->890219625],

Cell[TextData[{
 "For the mesh here, the off-diagonals are given by the sparse array ",
 StyleBox["M", "MR"],
 "."
}], "Text",
 CellID->402668726],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"num", " ", "=", " ", 
   RowBox[{"Max", "[", "edges", "]"}]}], ";"}], "\n", 
 RowBox[{"mat", "=", 
  RowBox[{"SparseArray", "[", 
   RowBox[{
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Rule", "[", 
        RowBox[{"#", ",", 
         RowBox[{"-", "1"}]}], "]"}], "&"}], ",", "edges"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"num", ",", "num"}], "}"}]}], "]"}]}]}], "Input",
 CellLabel->"In[9]:=",
 CellID->932306185],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "2482", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 2482, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"452", ",", "452"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{216, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->8915849]
}, Open  ]],

Cell["\<\
Each diagonal in a row is the negated sum of the off-diagonals. Thus the \
diagonal part of matrix is as follows.\
\>", "Text",
 CellID->960126094],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"diag", " ", "=", " ", 
  RowBox[{
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i_", ",", "i_"}], "}"}], "\[Rule]", "1."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"num", ",", "num"}], "}"}]}], "]"}], "  ", 
   RowBox[{"SparseArray", "[", " ", 
    RowBox[{"-", 
     RowBox[{"mat", ".", 
      RowBox[{"Table", "[", 
       RowBox[{"1.", ",", 
        RowBox[{"{", "num", "}"}]}], "]"}]}]}], "]"}]}]}]], "Input",
 CellLabel->"In[11]:=",
 CellID->647138300],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "452", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 452, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"452", ",", "452"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{209, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->125108758]
}, Open  ]],

Cell["Therefore, the Laplacian is given as below.", "Text",
 CellID->1298645120],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"lap", "=", 
  RowBox[{"mat", "+", "diag"}]}]], "Input",
 CellLabel->"In[12]:=",
 CellID->1452227379],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "2934", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 2934, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"452", ",", "452"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{216, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->790506248]
}, Open  ]],

Cell["This visualizes the matrix.", "Text",
 CellID->675614652],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatrixPlot", "[", "lap", "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->1309414475],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 174},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->542032939]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Fiedler Vector", "Subsection",
 CellTags->{"t:219", "c:165"},
 CellID->1726888774],

Cell["\<\
The sum of each row is zero, therefore the Laplacian matrix is positive \
semi-definite with a zero eigenvalue. The eigenvector corresponding to the \
second smallest eigenvalue is called the Fiedler vector.\
\>", "Text",
 CellID->578135413],

Cell["\<\
This calculates the two smallest eigenvalues and eigenvectors.\
\>", "Text",
 CellID->1505867263],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"lam2", ",", "lam1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"vec2", ",", "vec1"}], "}"}]}], "}"}], "=", 
   RowBox[{"Eigensystem", "[", 
    RowBox[{"lap", ",", 
     RowBox[{"-", "2"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->1016309346],

Cell["One of these eigenvalues is zero.", "Text",
 CellID->26249984],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{"{", 
   RowBox[{"lam1", ",", "lam2"}], "}"}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->949333453],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "0.013817879470281617`"}], "}"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->541658671]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Partitioning the Nodes", "Subsection",
 CellTags->{"t:220", "c:166"},
 CellID->1327004163],

Cell[TextData[{
 "Now use the Fiedler eigenvector to generate an ordering of vertices and \
form two groups of vertices: ",
 StyleBox["domain1", "MR"],
 " and ",
 StyleBox["domain2", "MR"],
 "."
}], "Text",
 CellID->1341293446],

Cell[BoxData[{
 RowBox[{
  RowBox[{"order", " ", "=", " ", 
   RowBox[{"Ordering", "[", "vec2", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"domain1", " ", "=", " ", 
   RowBox[{"Drop", "[", 
    RowBox[{"order", ",", 
     RowBox[{"Floor", "[", 
      RowBox[{"num", "/", "2"}], "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"domain2", "=", " ", 
   RowBox[{"Drop", "[", 
    RowBox[{"order", ",", 
     RowBox[{"-", 
      RowBox[{"(", 
       RowBox[{"num", "-", 
        RowBox[{"Floor", "[", 
         RowBox[{"num", "/", "2"}], "]"}]}], ")"}]}]}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[16]:=",
 CellID->983104363],

Cell[TextData[{
 "This sets the colors of vertices in ",
 StyleBox["domain1", "MR"],
 " as ",
 StyleBox["Hue[0.]", "MR"],
 " (red) and vertices in ",
 StyleBox["domain2", "MR"],
 " as ",
 StyleBox["Hue[0.75]", "MR"],
 " (blue)."
}], "Text",
 CellID->295848923],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"NodeColor", "[", "#", "]"}], " ", "=", " ", 
       RowBox[{"Hue", "[", "0.", "]"}]}], ")"}], "&"}], ",", "domain1"}], 
   "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Map", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"NodeColor", "[", "#", "]"}], " ", "=", " ", 
       RowBox[{"Hue", "[", "0.75", "]"}]}], ")"}], "&"}], ",", "domain2"}], 
   "]"}], ";"}]}], "Input",
 CellLabel->"In[19]:=",
 CellID->908206056],

Cell["\<\
This plots the partition. As can be seen, the partitions avoided the very \
dense part of the mesh, and each subdomain is nicely connected.\
\>", "Text",
 CellID->2052903963],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"graphicsEdge", "[", 
    RowBox[{"{", 
     RowBox[{"i_", ",", "j_"}], "}"}], "]"}], ":=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"NodeColor", "[", "i", "]"}], ",", 
     RowBox[{"Line", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"coordinates", "[", 
         RowBox[{"[", "i", "]"}], "]"}], ",", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"coordinates", "[", 
            RowBox[{"[", "j", "]"}], "]"}], "+", 
           RowBox[{"coordinates", "[", 
            RowBox[{"[", "i", "]"}], "]"}]}], ")"}], "/", "2."}]}], "}"}], 
      "]"}], ",", "\[IndentingNewLine]", 
     RowBox[{"NodeColor", "[", "j", "]"}], ",", 
     RowBox[{"Line", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"coordinates", "[", 
         RowBox[{"[", "j", "]"}], "]"}], ",", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"coordinates", "[", 
            RowBox[{"[", "j", "]"}], "]"}], "+", 
           RowBox[{"coordinates", "[", 
            RowBox[{"[", "i", "]"}], "]"}]}], ")"}], "/", "2."}]}], "}"}], 
      "]"}]}], "\[IndentingNewLine]", "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Lines", " ", "=", " ", 
   RowBox[{"Flatten", "[", 
    RowBox[{"Map", "[", 
     RowBox[{"graphicsEdge", ",", "edges"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Graphics", "[", "Lines", "]"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}]}], "]"}]}], "Input",
 CellLabel->"In[21]:=",
 CellID->1684633439],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->6541464]
}, Open  ]],

Cell["\<\
The number of edges cut by the partition can be calculated as follows.\
\>", "Text",
 CellID->854860779],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"vec", "=", 
    RowBox[{"Table", "[", 
     RowBox[{"1", ",", 
      RowBox[{"{", "num", "}"}]}], "]"}]}], ";"}], " "}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"vec", "[", 
     RowBox[{"[", "domain1", "]"}], "]"}], " ", "=", " ", "0"}], ";"}], 
  " "}], "\n", 
 RowBox[{
  RowBox[{"Total", "[", 
   RowBox[{"Abs", "[", 
    RowBox[{"lap", ".", "vec"}], "]"}], "]"}], "/", "2"}]}], "Input",
 CellLabel->"In[24]:=",
 CellID->979514223],

Cell[BoxData["55.`"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[26]=",
 CellID->18912229]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Matrix Functions with NDSolve", "Section",
 CellTags->{"t:117", "c:167"},
 CellID->747736829],

Cell[TextData[{
 "One of the important features of many of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 "'s numerical solvers is that they work for vector and matrix input. \
Typically this is used to solve coupled systems of equations, but it can be \
used to solve matrix specific problems. One example will be given here that \
uses the numerical differential equation solver, ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ", to find parametrized solutions to matrix functions."
}], "Text",
 CellID->138582759],

Cell["Here is a 3\[Cross]3 matrix.", "Text",
 CellID->1288157165],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"matA", "=", 
   RowBox[{"N", "[", 
    RowBox[{"-", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"4", ",", "5", ",", "6"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"7", ",", "8", ",", "9"}], "}"}]}], "}"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"matA", "//", "MatrixForm"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1274826748],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztVitzAjEQDvegL+Cgj+HVzuD7P1pVX1Fzw5ShotMOReCQSCSSn4BEIpFI
JBKJRKab3IbL7YQjTCsqysyR7N7m+3Y32ew9hd3263vYfWuGjcdO+Nl+a341
Hj46oHIzjLEWPPcVBnMOM/m0OIdR//fFvzsej7n8sWcp9/t9lF1p5cnpcrmc
TqdimpeyQIytHBBvYBRqD+VrlH2UTxXbYrGg6KPRCNH2oQs05u/0ci3bvS2T
VU6Cc7vd9nq9mNNV2ohW+iCgswSliHrXyF3CtydkVU7Kg8EACJQv6Ol8Pi8S
64oRuwajB2NBWkPiIf1JLJmz9XotpgHBrGLma0bsAtlPHTPee+pn2YhVTWGA
8KOjtYchdwSiWRuoPKAqs5fHx2yIgOo4FomdefXLvxVqA8ybqi1a71Q+t0KV
u+DpO3u42tPf2vGqmhf6LJ4GUzUHPO0OCKy4zCf9jjCZK8yOIZ/Monba1U1i
d9oLmk/qHtJ9pHLpV1DrRLbzVWTG0SJUdW2Xsb9aZcdYDYdDC6u4v+tWcGtu
NhvO4/tRry5dpnV9JtevVqvJZCKm+u0OLQW7u77ncsp/XtkR82w2M3RDCAb9
UZXtEw9Uv/aI/pKn9XHIcaKPH/peoWeZVnmO7FGM6kDmaNZuiVzDTAg7B+KV
3mCksU+5Az5QHy/IeldDjT4xTKjHdOo8wYoYrjAaEaXemeUpwG9UlvkG2wot
0Q==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{102, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]//MatrixForm=",
 CellID->107684457]
}, Open  ]],

Cell["\<\
The exponential function is the solution of the differential equation.\
\>", "Text",
 CellID->1498225422],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"x", "'"}], 
     RowBox[{"(", "t", ")"}]}], "\[LongEqual]", 
    RowBox[{"a", " ", 
     RowBox[{"x", "(", "t", ")"}]}]}], ",", "  ", 
   RowBox[{
    RowBox[{"x", "(", "0", ")"}], "\[LongEqual]", "1"}]}], 
  TraditionalForm]], "NumberedEquation",
 GridBoxOptions->{
 GridBoxAlignment->{
  "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
   "RowsIndexed" -> {}}},
 CellID->1428906479],

Cell["\<\
This can be demonstrated by solving the equation symbolically.\
\>", "Text",
 CellID->321568152],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"a", " ", 
       RowBox[{"x", "[", "t", "]"}]}]}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", "x", ",",
    "t"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1527074812],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"a", " ", "t"}]]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{177, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->12430288]
}, Open  ]],

Cell[TextData[{
 "The matrix exponential ",
 StyleBox["Exp[A t]", "MR"],
 " can be found by forming a matrix equation and solving with ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ". The solution here is valid for ",
 StyleBox["0\[LessEqual]t\[LessEqual]5", "MR"],
 "."
}], "Text",
 CellID->213935367],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"solExp", "=", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"X", "'"}], "[", "t", "]"}], "\[Equal]", 
       RowBox[{"matA", ".", 
        RowBox[{"X", "[", "t", "]"}]}]}], "&&", 
      RowBox[{
       RowBox[{"X", "[", "0", "]"}], "\[Equal]", 
       RowBox[{"IdentityMatrix", "[", "3", "]"}]}]}], ",", "X", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "5"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"matrixExpA", "=", 
  RowBox[{"X", "/.", 
   RowBox[{"First", "[", "solExp", "]"}]}]}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->724088091],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0.`", ",", "5.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{258, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->203014520]
}, Open  ]],

Cell[TextData[{
 "The solution is a function that will return the matrix function ",
 StyleBox["Exp[A t]", "MR"],
 " for a range of ",
 StyleBox["t", "MR"],
 ". If t is 0.0, the identity matrix is returned."
}], "Text",
 CellID->1956245244],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"matrixExpA", "[", "0.", "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->639180338],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztVjtPwzAQduIG8ShtER0Yu/MnGBBM9D9EFVI7IFDp0C1jxowZ8xMyduyY
MWPGjB07ZjTnV+26JhRSJJAaKfGdfb7vcg/7nvzZ+PnFn01G/uBx6r+NJ6P3
wcPrFKawgxAawnt7g4AmQLF3SAiM+tejX5wkCWEPumN8EASCx0yqxciiKBaL
BSUpjxGfd2G8NPg2jB6MWPCnEiXPc1NrHMc6Ft1G5IM0He3a1Q5TsFk1kKuq
ms/nChnLWQ7OLdnRfWWdvTaQOHIYhqCOku4Gwc2yzGusGVwOjt/WzLy2Wq0O
Y7uKq46gcqA5AriGp9gvIHSlPwTvWBFoHLDYLUdXYmhy9t33RynDj59VunkS
nO+l1R7/r2q7/kRogmzmmD0Tu0eEf4SgnwA0P/qC7+21+6/W43ekoijaQ0r1
AroUnK/r9ZqISNVVvMmfsf1lWaZpSkn9BoBrR3QEJ+QnNV6/ypGXy6Xl/oSf
EfaYyPYcs892pF+37n6zr/EaI6gYKQQXPNfEdhonF3zArBU2KptbDTXv9HUC
gTcth0FQujjCBdmubHnX99kq73eR8wED2Why\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{78, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]//MatrixForm=",
 CellID->42946923]
}, Open  ]],

Cell[TextData[{
 "This demonstrates that ",
 StyleBox["Exp[A].Exp[A]\[LongEqual]Exp[2A]", "MR"],
 "."
}], "Text",
 CellID->1900783713],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"expA", "=", 
   RowBox[{"matrixExpA", "[", "1.", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"expA", ".", "expA"}], "-", 
   RowBox[{"matrixExpA", "[", "2.", "]"}]}], "//", "MatrixForm"}]}], "Input",
 CellLabel->"In[7]:=",
 CellID->505131619],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {
      RowBox[{"-", "8.489269731626337`*^-8"}], 
      RowBox[{"-", "2.3019632422460745`*^-8"}], "3.885342536591452`*^-8"},
     {
      RowBox[{"-", "9.34782684591795`*^-9"}], 
      RowBox[{"-", "2.5119254454253337`*^-9"}], "4.32397717631261`*^-9"},
     {"6.619703718513392`*^-8", "1.7995781975699288`*^-8", 
      RowBox[{"-", "3.020547145737851`*^-8"}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$, BoxForm`opts$1039]]]], "Output",
 ImageSize->{319, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]//MatrixForm=",
 CellID->335026987]
}, Open  ]],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " already has a function for computing the exponential of a function, ",
 Cell[BoxData[
  ButtonBox["MatrixExp",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MatrixExp"]], "InlineFormula"],
 ". This can be compared with the solution that was computed."
}], "Text",
 CellID->1899313066],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"dat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"t", ",", 
       RowBox[{"Norm", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"MatrixExp", "[", 
           RowBox[{
            RowBox[{"N", "[", "matA", "]"}], ",", 
            RowBox[{"\"\<Parameter\>\"", "\[Rule]", "t"}]}], "]"}], "-", 
          RowBox[{"matrixExpA", "[", "t", "]"}]}], ",", "\"\<Frobenius\>\""}],
         "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0.0001", ",", "5", ",", "0.2"}], "}"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"ListLogPlot", "[", "dat", "]"}]}], "Input",
 CellLabel->"In[9]:=",
 CellID->1557771254],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWD1vE0EQXfsuEbgg56CgwNkhZarQoKRACCEErqAKQqLiFKEkhUmUpKJK
5SrubQk64A8QwQ+ggYqWmj9y7Med5vJuPLfW2YgCS7672X3v7czs7tzaT5PT
/df95PRgN1nvHSdH+we7J+tPDo91U9BQSl3o79Gq0s+pfrLfizTV9+J1S18D
3dXR91Df48y+bXrNQ2o/tl2pbWjtFGx9VWtC/2SVboV9K/Otw7JxTINuarOo
onK7kh0zETQnsi06JLucgfgSfyuz1rJ7R8x2ebZuAm7Eti5Z++OHnw/uj4fn
39TAom7o63sNagicH99/66avX36pEPoDsFs1+6M561fh+Szw+bwK+TwXUC+e
f9Kox4/eiSid58M3n02eeRTvxwr4cSagyA8JRX7wKN6PWbW6RZZ/mqnbE23A
bkDUfUsy6Ms7rwm8a6wO5aWeDmVuGh1jjjzxi1k+FtncVfMXSogFfhYCmoUg
G9PcDdVRChIljRhm563FtSf4HsMcOHRU6WkMOffljURv5ru6sXUTMrUnoChD
DrWaczXQcO9APooozdUowx0XWqmyj2cUn9/u9V2tuMv+VR6uIbPipt3VpV06
1Z7ENxu+qZ3C8kxQzntas12wI7BfpX770unSKu+CHYHtdNtZrnk12g1dsCOw
8X2+k3Kr/B6gAvBJYhEqgLElFqFesqh5VaYEIr0roCgyCUWR8Kj/1ca/WuRv
LtO8AzP10FOn+izQA+UWzLYbqd5ZoQeaIYyJeyWAfrtLPKpLD3RCGBdP6qQb
MKeEZ6CGJ3jHxpXI1/XlLGd8L678PYh/U0BRNBJqcg3EuYhZlRN2BsmWWDgf
ZPOsv3s660P8GwKK4nUov9rUh6gdd/racQaertTSoVjq6VBcvjr5ySmvbQOI
a8lT5wrYeD4aQJxOV6pkVf99DCDiFnhuPVTXK1SGwMLzhVOZ9NtlCGisncSW
qtIQWDaXFVXK/ROlGn8ALtjCdQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->360925134]
}, Open  ]],

Cell[TextData[{
 "Here is a differential equation for the ",
 Cell[BoxData[
  ButtonBox["Sin",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sin"]], "InlineFormula"],
 " function."
}], "Text",
 CellID->395494351],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{
   RowBox[{"{", "  ", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], "+", " ", 
       RowBox[{
        SuperscriptBox["a", "2"], " ", 
        RowBox[{"x", "[", "t", "]"}]}]}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], "\[Equal]", "0"}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "0", "]"}], "\[Equal]", "a"}]}], "}"}], ",", 
   "x", ",", "t"}], "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->611669792],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", "t", "}"}], ",", 
      RowBox[{"Sin", "[", 
       RowBox[{"a", " ", "t"}], "]"}]}], "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{210, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->469641567]
}, Open  ]],

Cell[TextData[{
 "This computes the ",
 Cell[BoxData[
  ButtonBox["Sin",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sin"]], "InlineFormula"],
 " of the matrix ",
 StyleBox["A", "MR"],
 "."
}], "Text",
 CellID->1103099131],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"solSin", "=", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"X", "''"}], "[", "t", "]"}], "+", 
        RowBox[{"matA", ".", "matA", ".", 
         RowBox[{"X", "[", "t", "]"}]}]}], "\[Equal]", "0"}], "&&", 
      RowBox[{
       RowBox[{"X", "[", "0", "]"}], "\[Equal]", 
       RowBox[{"DiagonalMatrix", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}], "&&", 
      RowBox[{
       RowBox[{
        RowBox[{"X", "'"}], "[", "0", "]"}], "\[Equal]", "matA"}]}], ",", "X",
      ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", 
       RowBox[{"4", "Pi"}]}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"matrixSinA", "=", 
  RowBox[{"X", "/.", 
   RowBox[{"First", "[", "solSin", "]"}]}]}]}], "Input",
 CellLabel->"In[12]:=",
 CellID->560770166],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0.`", ",", "12.566370614359172`"}], "}"}], "}"}], 
    ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{293, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->491353434]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Sin",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sin"]], "InlineFormula"],
 " of the matrix has the correct behavior at the origin."
}], "Text",
 CellID->627744185],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"matrixSinA", "[", "0.", "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->2056843231],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAw+AGxtgQDkP0fyAJj
v///gTQyyQoimefPn/8fDBgcwPz6+noonxmsigXMvH79+vbt20FMEJ+ZASLO
BKR5CfA5YLacP38e3dTp06cj28UKcicMANlgUZAZPBTIQuz//v17RUUFwn5m
mCjECRD3YJghSIIoH9jE9vZ2oKEgJhPcHqbjx4+zUsl8YCQAowLVfHA4Pn/+
nJr+QMQ3sj2ItEEte4CBBUmANLOHHxZCUCFGrPaA4geUaoXRaAE0ddh1u4yq
QgtHYksHLqJMxZ4KqFE6UGI/6alw1J6hbc9oKdHf30+EKkSbAlkVsAx+//49
cjgSW0pwgvXfv39//fr1ICZyXQGspqAtC9qVEhD79+/fj6XuBXoJ6ipq1YnA
MEZpQxDXViLdHkSsIexhAoYl5f4AxR8TMFTAJkHbmQj3M1HFfIwQgdoDaQhR
0x6EiRB7uP9jLwVEwLKQtjUDIwCHIZ48\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{78, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]//MatrixForm=",
 CellID->976158977]
}, Open  ]],

Cell[TextData[{
 "This computes the ",
 Cell[BoxData[
  ButtonBox["Cos",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Cos"]], "InlineFormula"],
 " of the matrix ",
 StyleBox["A", "MR"],
 "."
}], "Text",
 CellID->350004496],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"solCos", "=", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"X", "''"}], "[", "t", "]"}], "+", 
        RowBox[{"matA", ".", "matA", ".", 
         RowBox[{"X", "[", "t", "]"}]}]}], "\[Equal]", "0"}], "&&", 
      RowBox[{
       RowBox[{"X", "[", "0", "]"}], "\[Equal]", 
       RowBox[{"IdentityMatrix", "[", "3", "]"}]}], "&&", 
      RowBox[{
       RowBox[{
        RowBox[{"X", "'"}], "[", "0", "]"}], "\[Equal]", 
       RowBox[{"DiagonalMatrix", "[", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "0"}], "}"}], "]"}]}]}], ",", "X", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", 
       RowBox[{"4", "Pi"}]}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"matrixCosA", "=", 
  RowBox[{"X", "/.", 
   RowBox[{"First", "[", "solCos", "]"}]}]}]}], "Input",
 CellLabel->"In[15]:=",
 CellID->567855139],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0.`", ",", "12.566370614359172`"}], "}"}], "}"}], 
    ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{293, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->69415140]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Cos",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Cos"]], "InlineFormula"],
 " of the matrix has the correct behavior at the origin."
}], "Text",
 CellID->2082608635],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"matrixCosA", "[", "0.", "]"}], "//", "MatrixForm"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->567365992],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztVjtPwzAQduIG8ShtER0Yu/MnGBBM9D9EFVI7IFDp0C1jxowZ8xMyduyY
MWPGjB07ZjTnV+26JhRSJJAaKfGdfb7vcg/7nvzZ+PnFn01G/uBx6r+NJ6P3
wcPrFKawgxAawnt7g4AmQLF3SAiM+tejX5wkCWEPumN8EASCx0yqxciiKBaL
BSUpjxGfd2G8NPg2jB6MWPCnEiXPc1NrHMc6Ft1G5IM0He3a1Q5TsFk1kKuq
ms/nChnLWQ7OLdnRfWWdvTaQOHIYhqCOku4Gwc2yzGusGVwOjt/WzLy2Wq0O
Y7uKq46gcqA5AriGp9gvIHSlPwTvWBFoHLDYLUdXYmhy9t33RynDj59VunkS
nO+l1R7/r2q7/kRogmzmmD0Tu0eEf4SgnwA0P/qC7+21+6/W43ekoijaQ0r1
AroUnK/r9ZqISNVVvMmfsf1lWaZpSkn9BoBrR3QEJ+QnNV6/ypGXy6Xl/oSf
EfaYyPYcs892pF+37n6zr/EaI6gYKQQXPNfEdhonF3zArBU2KptbDTXv9HUC
gTcth0FQujjCBdmubHnX99kq73eR8wED2Why\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{78, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]//MatrixForm=",
 CellID->59903819]
}, Open  ]],

Cell[TextData[{
 "This demonstrates the identity ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"sin", "(", "x", ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{"cos", "(", "x", ")"}], 
      RowBox[{"2", " "}]]}], "\[LongEqual]", "1"}], TraditionalForm]]],
 ", for ",
 StyleBox["t", "MR"],
 " \[LongEqual] 3."
}], "Text",
 CellID->1407317659],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"mat1", "=", 
      RowBox[{"matrixCosA", "[", "3.", "]"}]}], ";", 
     RowBox[{"mat1", ".", "mat1"}]}], ")"}], "+", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"mat2", "=", 
      RowBox[{"matrixSinA", "[", "3.", "]"}]}], ";", 
     RowBox[{"mat2", ".", "mat2"}]}], ")"}]}], "//", "MatrixForm"}]], "Input",\

 CellLabel->"In[18]:=",
 CellID->403006755],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"0.9999999872892993`", 
      RowBox[{"-", "1.5177718110104266`*^-8"}], 
      RowBox[{"-", "1.7644760541557503`*^-8"}]},
     {
      RowBox[{"-", "2.7794373524958615`*^-8"}], "0.9999999658972836`", 
      RowBox[{"-", "4.041093865092549`*^-8"}]},
     {
      RowBox[{"-", "4.2878172457072594`*^-8"}], 
      RowBox[{"-", "5.3027974233899045`*^-8"}], "0.999999936822247`"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$, BoxForm`opts$1100]]]], "Output",
 ImageSize->{319, 51},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]//MatrixForm=",
 CellID->543886976]
}, Open  ]],

Cell[CellGroupData[{

Cell["MORE ABOUT", "TutorialMoreAboutSection",
 CellID->864034266],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "TutorialMoreAbout",
 CellID->4791866],

Cell[TextData[ButtonBox["Linear Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/LinearSystems"]], "TutorialMoreAbout",
 CellID->212289775]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->520107442],

Cell[TextData[{
 ButtonBox["Linear Algebra in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraOverview"],
  FontSlant->"Italic"]
}], "RelatedTutorials",
 CellID->315063104],

Cell[TextData[ButtonBox["Matrix and Tensor Operations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations"]], \
"RelatedTutorials",
 CellID->3571032],

Cell[TextData[ButtonBox["Working with Sparse Arrays",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/LinearAlgebraSparseArrays"]], "RelatedTutorials",\

 CellID->560755906],

Cell[TextData[ButtonBox["Matrix Computations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/LinearAlgebraMatrixComputations"]], "RelatedTutorials",
 CellID->335456077],

Cell[TextData[ButtonBox["Matrix Types",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/LinearAlgebraMatrixTypes"]], "RelatedTutorials",\

 CellID->472696132]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Linear Algebra Examples - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 38, 31.5849796}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "This tutorial shows a number of examples of the use of Mathematica for \
computations that involve linear algebra. Certain sparse matrix techniques \
try to reorder the matrix so that elements are grouped into blocks. The \
computation then works on each block using dense matrix techniques. One \
simple way to order a matrix into blocks involves sorting according to the \
sum of elements on each row. This will be demonstrated in this example. \
First, generate a symmetric random sparse matrix.", "synonyms" -> {}, "title" -> 
    "Linear Algebra Examples", "type" -> "Tutorial", "uri" -> 
    "tutorial/LinearAlgebraExamples"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "t:58"->{
  Cell[2706, 74, 114, 3, 70, "Title",
   CellTags->{"t:58", "c:152"},
   CellID->349852864]},
 "c:152"->{
  Cell[2706, 74, 114, 3, 70, "Title",
   CellTags->{"t:58", "c:152"},
   CellID->349852864]},
 "t:151"->{
  Cell[3054, 89, 86, 2, 70, "Section",
   CellTags->{"t:151", "c:153"},
   CellID->1945860599]},
 "c:153"->{
  Cell[3054, 89, 86, 2, 70, "Section",
   CellTags->{"t:151", "c:153"},
   CellID->1945860599]},
 "t:191"->{
  Cell[20466, 495, 103, 2, 70, "Section",
   CellTags->{"t:191", "c:154"},
   CellID->967762464]},
 "c:154"->{
  Cell[20466, 495, 103, 2, 70, "Section",
   CellTags->{"t:191", "c:154"},
   CellID->967762464]},
 "t:105"->{
  Cell[22040, 545, 95, 2, 70, "Subsection",
   CellTags->{"t:105", "c:155"},
   CellID->899751076]},
 "c:155"->{
  Cell[22040, 545, 95, 2, 70, "Subsection",
   CellTags->{"t:105", "c:155"},
   CellID->899751076]},
 "t:198"->{
  Cell[27603, 761, 89, 2, 70, "Subsection",
   CellTags->{"t:198", "c:156"},
   CellID->701643607]},
 "c:156"->{
  Cell[27603, 761, 89, 2, 70, "Subsection",
   CellTags->{"t:198", "c:156"},
   CellID->701643607]},
 "t:193"->{
  Cell[32716, 956, 106, 2, 70, "Section",
   CellTags->{"t:193", "c:157"},
   CellID->601898483]},
 "c:157"->{
  Cell[32716, 956, 106, 2, 70, "Section",
   CellTags->{"t:193", "c:157"},
   CellID->601898483]},
 "t:213"->{
  Cell[35771, 1046, 94, 2, 70, "Subsection",
   CellTags->{"t:213", "c:158"},
   CellID->526704053]},
 "c:158"->{
  Cell[35771, 1046, 94, 2, 70, "Subsection",
   CellTags->{"t:213", "c:158"},
   CellID->526704053]},
 "t:214"->{
  Cell[41322, 1245, 100, 2, 70, "Subsection",
   CellTags->{"t:214", "c:159"},
   CellID->1503089771]},
 "c:159"->{
  Cell[41322, 1245, 100, 2, 70, "Subsection",
   CellTags->{"t:214", "c:159"},
   CellID->1503089771]},
 "t:115"->{
  Cell[46762, 1439, 98, 2, 70, "Section",
   CellTags->{"t:115", "c:160"},
   CellID->1711482715]},
 "c:160"->{
  Cell[46762, 1439, 98, 2, 70, "Section",
   CellTags->{"t:115", "c:160"},
   CellID->1711482715]},
 "t:215"->{
  Cell[386062, 7383, 88, 2, 70, "Section",
   CellTags->{"t:215", "c:161"},
   CellID->1370044451]},
 "c:161"->{
  Cell[386062, 7383, 88, 2, 70, "Section",
   CellTags->{"t:215", "c:161"},
   CellID->1370044451]},
 "t:216"->{
  Cell[386917, 7402, 81, 2, 70, "Subsubsection",
   CellTags->{"t:216", "c:162"},
   CellID->1594202747]},
 "c:162"->{
  Cell[386917, 7402, 81, 2, 70, "Subsubsection",
   CellTags->{"t:216", "c:162"},
   CellID->1594202747]},
 "t:217"->{
  Cell[497296, 10366, 90, 2, 70, "Subsection",
   CellTags->{"t:217", "c:163"},
   CellID->846283239]},
 "c:163"->{
  Cell[497296, 10366, 90, 2, 70, "Subsection",
   CellTags->{"t:217", "c:163"},
   CellID->846283239]},
 "t:218"->{
  Cell[526889, 10896, 86, 2, 70, "Subsection",
   CellTags->{"t:218", "c:164"},
   CellID->455167669]},
 "c:164"->{
  Cell[526889, 10896, 86, 2, 70, "Subsection",
   CellTags->{"t:218", "c:164"},
   CellID->455167669]},
 "t:219"->{
  Cell[537179, 11186, 92, 2, 70, "Subsection",
   CellTags->{"t:219", "c:165"},
   CellID->1726888774]},
 "c:165"->{
  Cell[537179, 11186, 92, 2, 70, "Subsection",
   CellTags->{"t:219", "c:165"},
   CellID->1726888774]},
 "t:220"->{
  Cell[538531, 11242, 96, 2, 70, "Subsection",
   CellTags->{"t:220", "c:166"},
   CellID->1327004163]},
 "c:166"->{
  Cell[538531, 11242, 96, 2, 70, "Subsection",
   CellTags->{"t:220", "c:166"},
   CellID->1327004163]},
 "t:117"->{
  Cell[575883, 11955, 99, 2, 70, "Section",
   CellTags->{"t:117", "c:167"},
   CellID->747736829]},
 "c:167"->{
  Cell[575883, 11955, 99, 2, 70, "Section",
   CellTags->{"t:117", "c:167"},
   CellID->747736829]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"t:58", 598179, 12703},
 {"c:152", 598285, 12707},
 {"t:151", 598391, 12711},
 {"c:153", 598500, 12715},
 {"t:191", 598609, 12719},
 {"c:154", 598720, 12723},
 {"t:105", 598831, 12727},
 {"c:155", 598944, 12731},
 {"t:198", 599057, 12735},
 {"c:156", 599170, 12739},
 {"t:193", 599283, 12743},
 {"c:157", 599394, 12747},
 {"t:213", 599505, 12751},
 {"c:158", 599619, 12755},
 {"t:214", 599733, 12759},
 {"c:159", 599849, 12763},
 {"t:115", 599965, 12767},
 {"c:160", 600077, 12771},
 {"t:215", 600189, 12775},
 {"c:161", 600302, 12779},
 {"t:216", 600415, 12783},
 {"c:162", 600534, 12787},
 {"t:217", 600653, 12791},
 {"c:163", 600769, 12795},
 {"t:218", 600885, 12799},
 {"c:164", 601001, 12803},
 {"t:219", 601117, 12807},
 {"c:165", 601234, 12811},
 {"t:220", 601351, 12815},
 {"c:166", 601468, 12819},
 {"t:117", 601585, 12823},
 {"c:167", 601698, 12827}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 2081, 47, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2706, 74, 114, 3, 70, "Title",
 CellTags->{"t:58", "c:152"},
 CellID->349852864],
Cell[2823, 79, 206, 6, 70, "Text",
 CellID->166859762],
Cell[CellGroupData[{
Cell[3054, 89, 86, 2, 70, "Section",
 CellTags->{"t:151", "c:153"},
 CellID->1945860599],
Cell[3143, 93, 374, 7, 70, "Text",
 CellID->50627230],
Cell[3520, 102, 85, 1, 70, "Text",
 CellID->488330398],
Cell[3608, 105, 798, 26, 70, "Input",
 CellID->1833983999],
Cell[4409, 133, 65, 1, 70, "Text",
 CellID->998430637],
Cell[CellGroupData[{
Cell[4499, 138, 107, 3, 28, "Input",
 CellID->186498401],
Cell[4609, 143, 1245, 24, 205, "Output",
 Evaluatable->False,
 CellID->346437503]
}, Open  ]],
Cell[5869, 170, 306, 9, 70, "Text",
 CellID->343119060],
Cell[6178, 181, 453, 15, 70, "Input",
 CellID->808449225],
Cell[6634, 198, 127, 4, 70, "Text",
 CellID->477090006],
Cell[CellGroupData[{
Cell[6786, 206, 115, 3, 28, "Input",
 CellID->1895276591],
Cell[6904, 211, 1209, 24, 205, "Output",
 Evaluatable->False,
 CellID->638786328]
}, Open  ]],
Cell[8128, 238, 117, 3, 70, "Text",
 CellID->1502509129],
Cell[8248, 243, 1195, 39, 70, "Input",
 CellID->410726030],
Cell[9446, 284, 75, 1, 70, "Text",
 CellID->1059944752],
Cell[CellGroupData[{
Cell[9546, 289, 114, 3, 28, "Input",
 CellID->1114594999],
Cell[9663, 294, 7460, 126, 171, "Output",
 Evaluatable->False,
 CellID->407509172]
}, Open  ]],
Cell[17138, 423, 141, 4, 70, "Text",
 CellID->1812249831],
Cell[CellGroupData[{
Cell[17304, 431, 115, 3, 28, "Input",
 CellID->722344216],
Cell[17422, 436, 2995, 53, 171, "Output",
 Evaluatable->False,
 CellID->406068805]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[20466, 495, 103, 2, 70, "Section",
 CellTags->{"t:191", "c:154"},
 CellID->967762464],
Cell[20572, 499, 1274, 36, 70, "Text",
 CellID->142590070],
Cell[21849, 537, 166, 4, 70, "Text",
 CellID->2067040085],
Cell[CellGroupData[{
Cell[22040, 545, 95, 2, 70, "Subsection",
 CellTags->{"t:105", "c:155"},
 CellID->899751076],
Cell[22138, 549, 379, 10, 70, "Text",
 CellID->1683975805],
Cell[22520, 561, 616, 22, 70, "NumberedEquation",
 CellID->894370822],
Cell[23139, 585, 160, 6, 70, "Text",
 CellID->1452360001],
Cell[23302, 593, 1114, 28, 70, "Input",
 CellID->1515405005],
Cell[24419, 623, 75, 1, 70, "Text",
 CellID->255865615],
Cell[CellGroupData[{
Cell[24519, 628, 561, 19, 70, "Input",
 CellID->1071130011],
Cell[25083, 649, 310, 11, 36, "Output",
 CellID->511228008]
}, Open  ]],
Cell[25408, 663, 85, 1, 70, "Text",
 CellID->989357155],
Cell[CellGroupData[{
Cell[25518, 668, 97, 3, 70, "Input",
 CellID->159825364],
Cell[25618, 673, 270, 9, 36, "Output",
 CellID->924913556]
}, Open  ]],
Cell[25903, 685, 227, 8, 70, "Text",
 CellID->1867744758],
Cell[CellGroupData[{
Cell[26155, 697, 137, 4, 70, "Input",
 CellID->1417679507],
Cell[26295, 703, 259, 9, 36, "Output",
 CellID->137724080]
}, Open  ]],
Cell[26569, 715, 223, 8, 70, "Text",
 CellID->1388722621],
Cell[26795, 725, 121, 3, 70, "Text",
 CellID->1408815551],
Cell[CellGroupData[{
Cell[26941, 732, 333, 12, 70, "Input",
 CellID->1955757482],
Cell[27277, 746, 277, 9, 36, "Output",
 CellID->822716328]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[27603, 761, 89, 2, 70, "Subsection",
 CellTags->{"t:198", "c:156"},
 CellID->701643607],
Cell[27695, 765, 327, 10, 70, "Text",
 CellID->691397172],
Cell[28025, 777, 304, 9, 70, "NumberedEquation",
 CellID->278255887],
Cell[28332, 788, 152, 6, 70, "Text",
 CellID->1574793290],
Cell[28487, 796, 799, 24, 70, "Input",
 CellID->1962359402],
Cell[29289, 822, 115, 3, 70, "Text",
 CellID->320618788],
Cell[CellGroupData[{
Cell[29429, 829, 554, 19, 70, "Input",
 CellID->603499593],
Cell[29986, 850, 259, 9, 36, "Output",
 CellID->108077978]
}, Open  ]],
Cell[30260, 862, 225, 5, 70, "Text",
 CellID->1306789010],
Cell[30488, 869, 1157, 36, 70, "Input",
 CellID->887633201],
Cell[31648, 907, 108, 3, 70, "Text",
 CellID->2029797145],
Cell[CellGroupData[{
Cell[31781, 914, 155, 5, 70, "Input",
 CellID->2022883420],
Cell[31939, 921, 175, 5, 36, "Output",
 CellID->604937814]
}, Open  ]],
Cell[32129, 929, 167, 4, 70, "Text",
 CellID->1585451910],
Cell[CellGroupData[{
Cell[32321, 937, 155, 5, 70, "Input",
 CellID->713159394],
Cell[32479, 944, 176, 5, 36, "Output",
 CellID->480493485]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[32716, 956, 106, 2, 70, "Section",
 CellTags->{"t:193", "c:157"},
 CellID->601898483],
Cell[32825, 960, 525, 16, 70, "Text",
 CellID->1459826411],
Cell[33353, 978, 311, 10, 70, "NumberedEquation",
 CellID->301822702],
Cell[33667, 990, 358, 7, 70, "Text",
 CellID->377778300],
Cell[34028, 999, 556, 9, 70, "Text",
 CellID->1358227568],
Cell[34587, 1010, 718, 18, 70, "Text",
 CellID->2127051413],
Cell[35308, 1030, 289, 6, 70, "Text",
 CellID->192826682],
Cell[35600, 1038, 146, 4, 70, "Text",
 CellID->285221051],
Cell[CellGroupData[{
Cell[35771, 1046, 94, 2, 70, "Subsection",
 CellTags->{"t:213", "c:158"},
 CellID->526704053],
Cell[35868, 1050, 188, 6, 70, "Text",
 CellID->387425837],
Cell[36059, 1058, 547, 17, 70, "NumberedEquation",
 CellID->295624755],
Cell[36609, 1077, 170, 6, 70, "Text",
 CellID->1435581774],
Cell[36782, 1085, 480, 16, 70, "NumberedEquation",
 CellID->844101573],
Cell[37265, 1103, 83, 1, 70, "Text",
 CellID->538165329],
Cell[37351, 1106, 2316, 65, 70, "Input",
 CellID->1891046986],
Cell[39670, 1173, 59, 1, 70, "Text",
 CellID->68718102],
Cell[CellGroupData[{
Cell[39754, 1178, 558, 19, 70, "Input",
 CellID->1746421875],
Cell[40315, 1199, 229, 8, 36, "Output",
 CellID->334768554]
}, Open  ]],
Cell[40559, 1210, 127, 4, 70, "Text",
 CellID->1293517785],
Cell[CellGroupData[{
Cell[40711, 1218, 333, 12, 70, "Input",
 CellID->1176453914],
Cell[41047, 1232, 226, 7, 36, "Output",
 CellID->390691631]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[41322, 1245, 100, 2, 70, "Subsection",
 CellTags->{"t:214", "c:159"},
 CellID->1503089771],
Cell[41425, 1249, 241, 7, 70, "Text",
 CellID->1579641536],
Cell[41669, 1258, 304, 8, 70, "NumberedEquation",
 CellID->1842451338],
Cell[41976, 1268, 170, 6, 70, "Text",
 CellID->1512931673],
Cell[42149, 1276, 512, 16, 70, "NumberedEquation",
 CellID->1636247631],
Cell[42664, 1294, 84, 1, 70, "Text",
 CellID->1885834382],
Cell[42751, 1297, 2252, 64, 70, "Input",
 CellID->507312201],
Cell[45006, 1363, 60, 1, 70, "Text",
 CellID->886083617],
Cell[CellGroupData[{
Cell[45091, 1368, 565, 19, 70, "Input",
 CellID->1755742621],
Cell[45659, 1389, 261, 9, 36, "Output",
 CellID->335256515]
}, Open  ]],
Cell[45935, 1401, 128, 4, 70, "Text",
 CellID->1965002129],
Cell[CellGroupData[{
Cell[46088, 1409, 333, 12, 70, "Input",
 CellID->1787258537],
Cell[46424, 1423, 277, 9, 36, "Output",
 CellID->189879261]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[46762, 1439, 98, 2, 70, "Section",
 CellTags->{"t:115", "c:160"},
 CellID->1711482715],
Cell[46863, 1443, 275, 6, 70, "Text",
 CellID->1494023087],
Cell[47141, 1451, 985, 29, 70, "NumberedEquation",
 CellID->1667222485],
Cell[48129, 1482, 304, 9, 70, "Text",
 CellID->641679339],
Cell[CellGroupData[{
Cell[48458, 1495, 618, 21, 65, "Input",
 CellID->41775339],
Cell[49079, 1518, 2313, 42, 88, "Output",
 Evaluatable->False,
 CellID->130715566]
}, Open  ]],
Cell[51407, 1563, 113, 3, 70, "Text",
 CellID->1483891766],
Cell[51523, 1568, 447, 15, 70, "Input",
 CellID->1041318551],
Cell[51973, 1585, 271, 5, 70, "Text",
 CellID->1596647254],
Cell[52247, 1592, 258, 8, 70, "Input",
 CellID->2042228713],
Cell[52508, 1602, 93, 1, 70, "Text",
 CellID->506849390],
Cell[CellGroupData[{
Cell[52626, 1607, 872, 27, 49, "Input",
 CellID->1697553728],
Cell[53501, 1636, 3192, 56, 122, "Output",
 Evaluatable->False,
 CellID->276418174]
}, Open  ]],
Cell[56708, 1695, 640, 14, 70, "Text",
 CellID->1364309358],
Cell[CellGroupData[{
Cell[57373, 1713, 259, 8, 70, "Input",
 CellID->1264480887],
Cell[57635, 1723, 886, 30, 36, "Output",
 CellID->31349214]
}, Open  ]],
Cell[58536, 1756, 221, 8, 70, "Text",
 CellID->54307570],
Cell[CellGroupData[{
Cell[58782, 1768, 233, 8, 70, "Input",
 CellID->66779015],
Cell[59018, 1778, 224, 7, 36, "Output",
 CellID->11356225]
}, Open  ]],
Cell[59257, 1788, 110, 3, 70, "Text",
 CellID->1991746674],
Cell[59370, 1793, 178, 6, 70, "Input",
 CellID->311260482],
Cell[59551, 1801, 77, 1, 70, "Text",
 CellID->84756882],
Cell[CellGroupData[{
Cell[59653, 1806, 181, 5, 70, "Input",
 CellID->727257813],
Cell[59837, 1813, 150390, 2501, 70, "Output",
 CellID->43200405]
}, Open  ]],
Cell[210242, 4317, 509, 12, 70, "Text",
 CellID->283159489],
Cell[CellGroupData[{
Cell[210776, 4333, 149, 4, 70, "Input",
 CellID->237769871],
Cell[210928, 4339, 333, 12, 36, "Output",
 CellID->56876998]
}, Open  ]],
Cell[211276, 4354, 363, 13, 70, "Text",
 CellID->1972304745],
Cell[211642, 4369, 812, 27, 70, "Input",
 CellID->2115757770],
Cell[212457, 4398, 119, 3, 70, "Text",
 CellID->1506613364],
Cell[CellGroupData[{
Cell[212601, 4405, 181, 5, 70, "Input",
 CellID->671126971],
Cell[212785, 4412, 150789, 2511, 70, "Output",
 CellID->410111908]
}, Open  ]],
Cell[363589, 6926, 933, 25, 70, "Text",
 CellID->705583400],
Cell[CellGroupData[{
Cell[364547, 6955, 1290, 40, 70, "Input",
 CellID->335505066],
Cell[365840, 6997, 452, 17, 36, "Output",
 CellID->335257658]
}, Open  ]],
Cell[366307, 7017, 203, 8, 70, "Text",
 CellID->628068116],
Cell[CellGroupData[{
Cell[366535, 7029, 312, 10, 28, "Input",
 CellID->1058700607],
Cell[366850, 7041, 17902, 298, 154, "Output",
 Evaluatable->False,
 CellID->5905303]
}, Open  ]],
Cell[384767, 7342, 107, 3, 70, "Text",
 CellID->323427151],
Cell[CellGroupData[{
Cell[384899, 7349, 256, 8, 70, "Input",
 CellID->559016300],
Cell[385158, 7359, 855, 18, 75, "Output",
 CellID->82396169]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[386062, 7383, 88, 2, 70, "Section",
 CellTags->{"t:215", "c:161"},
 CellID->1370044451],
Cell[386153, 7387, 739, 11, 70, "Text",
 CellID->1017424823],
Cell[CellGroupData[{
Cell[386917, 7402, 81, 2, 70, "Subsubsection",
 CellTags->{"t:216", "c:162"},
 CellID->1594202747],
Cell[387001, 7406, 89, 1, 70, "Text",
 CellID->1759588857],
Cell[387093, 7409, 46573, 1363, 70, "Input",
 CellID->412616762],
Cell[433669, 8774, 84, 1, 70, "Text",
 CellID->625222771],
Cell[433756, 8777, 63503, 1584, 70, "Input",
 CellID->481627426]
}, Open  ]],
Cell[CellGroupData[{
Cell[497296, 10366, 90, 2, 70, "Subsection",
 CellTags->{"t:217", "c:163"},
 CellID->846283239],
Cell[497389, 10370, 88, 1, 70, "Text",
 CellID->698203792],
Cell[CellGroupData[{
Cell[497502, 10375, 1729, 56, 119, "Input",
 CellID->860484936],
Cell[499234, 10433, 27606, 457, 201, "Output",
 Evaluatable->False,
 CellID->106155769]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[526889, 10896, 86, 2, 70, "Subsection",
 CellTags->{"t:218", "c:164"},
 CellID->455167669],
Cell[526978, 10900, 894, 29, 70, "Text",
 CellID->666262269],
Cell[527875, 10931, 646, 21, 70, "NumberedEquation",
 CellID->890219625],
Cell[528524, 10954, 146, 5, 70, "Text",
 CellID->402668726],
Cell[CellGroupData[{
Cell[528695, 10963, 485, 16, 70, "Input",
 CellID->932306185],
Cell[529183, 10981, 488, 17, 36, "Output",
 CellID->8915849]
}, Open  ]],
Cell[529686, 11001, 157, 4, 70, "Text",
 CellID->960126094],
Cell[CellGroupData[{
Cell[529868, 11009, 558, 18, 70, "Input",
 CellID->647138300],
Cell[530429, 11029, 488, 17, 36, "Output",
 CellID->125108758]
}, Open  ]],
Cell[530932, 11049, 80, 1, 70, "Text",
 CellID->1298645120],
Cell[CellGroupData[{
Cell[531037, 11054, 124, 4, 70, "Input",
 CellID->1452227379],
Cell[531164, 11060, 490, 17, 36, "Output",
 CellID->790506248]
}, Open  ]],
Cell[531669, 11080, 63, 1, 70, "Text",
 CellID->675614652],
Cell[CellGroupData[{
Cell[531757, 11085, 110, 3, 28, "Input",
 CellID->1309414475],
Cell[531870, 11090, 5260, 90, 195, "Output",
 Evaluatable->False,
 CellID->542032939]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[537179, 11186, 92, 2, 70, "Subsection",
 CellTags->{"t:219", "c:165"},
 CellID->1726888774],
Cell[537274, 11190, 251, 5, 70, "Text",
 CellID->578135413],
Cell[537528, 11197, 107, 3, 70, "Text",
 CellID->1505867263],
Cell[537638, 11202, 366, 13, 70, "Input",
 CellID->1016309346],
Cell[538007, 11217, 68, 1, 70, "Text",
 CellID->26249984],
Cell[CellGroupData[{
Cell[538100, 11222, 154, 5, 70, "Input",
 CellID->949333453],
Cell[538257, 11229, 225, 7, 36, "Output",
 CellID->541658671]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[538531, 11242, 96, 2, 70, "Subsection",
 CellTags->{"t:220", "c:166"},
 CellID->1327004163],
Cell[538630, 11246, 227, 8, 70, "Text",
 CellID->1341293446],
Cell[538860, 11256, 632, 21, 70, "Input",
 CellID->983104363],
Cell[539495, 11279, 260, 11, 70, "Text",
 CellID->295848923],
Cell[539758, 11292, 568, 20, 70, "Input",
 CellID->908206056],
Cell[540329, 11314, 184, 4, 70, "Text",
 CellID->2052903963],
Cell[CellGroupData[{
Cell[540538, 11322, 1565, 46, 137, "Input",
 CellID->1684633439],
Cell[542106, 11370, 32903, 543, 201, "Output",
 Evaluatable->False,
 CellID->6541464]
}, Open  ]],
Cell[575024, 11916, 114, 3, 70, "Text",
 CellID->854860779],
Cell[CellGroupData[{
Cell[575163, 11923, 495, 18, 70, "Input",
 CellID->979514223],
Cell[575661, 11943, 161, 5, 36, "Output",
 CellID->18912229]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[575883, 11955, 99, 2, 70, "Section",
 CellTags->{"t:117", "c:167"},
 CellID->747736829],
Cell[575985, 11959, 596, 14, 70, "Text",
 CellID->138582759],
Cell[576584, 11975, 65, 1, 70, "Text",
 CellID->1288157165],
Cell[CellGroupData[{
Cell[576674, 11980, 478, 16, 47, "Input",
 CellID->1274826748],
Cell[577155, 11998, 960, 20, 78, "Output",
 Evaluatable->False,
 CellID->107684457]
}, Open  ]],
Cell[578130, 12021, 115, 3, 70, "Text",
 CellID->1498225422],
Cell[578248, 12026, 482, 16, 70, "NumberedEquation",
 CellID->1428906479],
Cell[578733, 12044, 106, 3, 70, "Text",
 CellID->321568152],
Cell[CellGroupData[{
Cell[578864, 12051, 402, 14, 70, "Input",
 CellID->1527074812],
Cell[579269, 12067, 393, 13, 39, "Output",
 CellID->12430288]
}, Open  ]],
Cell[579677, 12083, 377, 12, 70, "Text",
 CellID->213935367],
Cell[CellGroupData[{
Cell[580079, 12099, 650, 20, 70, "Input",
 CellID->724088091],
Cell[580732, 12121, 360, 13, 36, "Output",
 CellID->203014520]
}, Open  ]],
Cell[581107, 12137, 240, 7, 70, "Text",
 CellID->1956245244],
Cell[CellGroupData[{
Cell[581372, 12148, 140, 4, 28, "Input",
 CellID->639180338],
Cell[581515, 12154, 807, 17, 78, "Output",
 Evaluatable->False,
 CellID->42946923]
}, Open  ]],
Cell[582337, 12174, 134, 5, 70, "Text",
 CellID->1900783713],
Cell[CellGroupData[{
Cell[582496, 12183, 289, 9, 70, "Input",
 CellID->505131619],
Cell[582788, 12194, 1077, 28, 86, "Output",
 CellID->335026987]
}, Open  ]],
Cell[583880, 12225, 355, 10, 70, "Text",
 CellID->1899313066],
Cell[CellGroupData[{
Cell[584260, 12239, 696, 21, 101, "Input",
 CellID->1557771254],
Cell[584959, 12262, 1218, 24, 133, "Output",
 Evaluatable->False,
 CellID->360925134]
}, Open  ]],
Cell[586192, 12289, 215, 8, 70, "Text",
 CellID->395494351],
Cell[CellGroupData[{
Cell[586432, 12301, 573, 19, 70, "Input",
 CellID->611669792],
Cell[587008, 12322, 387, 13, 36, "Output",
 CellID->469641567]
}, Open  ]],
Cell[587410, 12338, 227, 10, 70, "Text",
 CellID->1103099131],
Cell[CellGroupData[{
Cell[587662, 12352, 901, 28, 70, "Input",
 CellID->560770166],
Cell[588566, 12382, 382, 14, 36, "Output",
 CellID->491353434]
}, Open  ]],
Cell[588963, 12399, 223, 8, 70, "Text",
 CellID->627744185],
Cell[CellGroupData[{
Cell[589211, 12411, 142, 4, 28, "Input",
 CellID->2056843231],
Cell[589356, 12417, 744, 16, 78, "Output",
 Evaluatable->False,
 CellID->976158977]
}, Open  ]],
Cell[590115, 12436, 226, 10, 70, "Text",
 CellID->350004496],
Cell[CellGroupData[{
Cell[590366, 12450, 938, 28, 70, "Input",
 CellID->567855139],
Cell[591307, 12480, 381, 14, 36, "Output",
 CellID->69415140]
}, Open  ]],
Cell[591703, 12497, 224, 8, 70, "Text",
 CellID->2082608635],
Cell[CellGroupData[{
Cell[591952, 12509, 141, 4, 28, "Input",
 CellID->567365992],
Cell[592096, 12515, 808, 17, 78, "Output",
 Evaluatable->False,
 CellID->59903819]
}, Open  ]],
Cell[592919, 12535, 391, 15, 70, "Text",
 CellID->1407317659],
Cell[CellGroupData[{
Cell[593335, 12554, 432, 15, 70, "Input",
 CellID->403006755],
Cell[593770, 12571, 1094, 29, 86, "Output",
 CellID->543886976]
}, Open  ]],
Cell[CellGroupData[{
Cell[594901, 12605, 66, 1, 70, "TutorialMoreAboutSection",
 CellID->864034266],
Cell[594970, 12608, 169, 3, 70, "TutorialMoreAbout",
 CellID->4791866],
Cell[595142, 12613, 147, 3, 70, "TutorialMoreAbout",
 CellID->212289775]
}, Open  ]],
Cell[CellGroupData[{
Cell[595326, 12621, 72, 1, 70, "RelatedTutorialsSection",
 CellID->520107442],
Cell[595401, 12624, 304, 9, 70, "RelatedTutorials",
 CellID->315063104],
Cell[595708, 12635, 191, 5, 70, "RelatedTutorials",
 CellID->3571032],
Cell[595902, 12642, 175, 4, 70, "RelatedTutorials",
 CellID->560755906],
Cell[596080, 12648, 175, 4, 70, "RelatedTutorials",
 CellID->335456077],
Cell[596258, 12654, 160, 4, 70, "RelatedTutorials",
 CellID->472696132]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[596457, 12663, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

