(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    111590,       4100]
NotebookOptionsPosition[     82697,       3075]
NotebookOutlinePosition[     91933,       3382]
CellTagsIndexPosition[     90786,       3334]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Linear Algebra in Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/LinearAlgebraOverview"], 
          "Matrix and Tensor Operations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations"], 
          "Working with Sparse Arrays" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraSparseArrays"], 
          "Matrix Computations" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraMatrixComputations"], "Matrix Types" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/LinearAlgebraMatrixTypes"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"Linear Algebra in Mathematica\"\>", 
       2->"\<\"Matrix and Tensor Operations\"\>", 
       3->"\<\"Working with Sparse Arrays\"\>", 
       4->"\<\"Matrix Computations\"\>", 
       5->"\<\"Matrix Types\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Matrices and Linear Algebra" :> 
          Documentation`HelpLookup["paclet:guide/MatricesAndLinearAlgebra"], 
          "Linear Systems" :> 
          Documentation`HelpLookup["paclet:guide/LinearSystems"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Matrices and Linear Algebra\"\>", 
       2->"\<\"Linear Systems\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"AbsoluteTiming" :> 
          Documentation`HelpLookup["paclet:ref/AbsoluteTiming"], 
          "TimeConstrained" :> 
          Documentation`HelpLookup["paclet:ref/TimeConstrained"], "Timing" :> 
          Documentation`HelpLookup["paclet:ref/Timing"], "True" :> 
          Documentation`HelpLookup["paclet:ref/True"]}, #, 2]& , Evaluator -> 
        Automatic], {1->"\<\"AbsoluteTiming\"\>", 
       2->"\<\"TimeConstrained\"\>", 3->"\<\"Timing\"\>", 
       4->"\<\"True\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Performance of Linear Algebra Computation", "Title",
 ShowCellBracket->True,
 CellTags->{"t:59", "c:131"},
 CellID->195254286],

Cell["This tutorial covers issues related to performance.", "Text",
 CellID->283772186],

Cell[CellGroupData[{

Cell["Packed Arrays", "Section",
 CellTags->{"t:109", "c:132"},
 CellID->1945433762],

Cell[TextData[{
 "One of the challenges of building ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " is to make sure that the system is general enough to support symbolic \
computation and fast enough so that numerical computation is efficient. These \
and other goals are described under \"",
 ButtonBox["Design Principles of ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraIntroduction#207795216"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraIntroduction#207795216"],
  FontSlant->"Italic"],
 "\". One technique that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " provides is to use specialized representations for important types of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expression. In the case of linear algebra this involves what is known as \
Packed Array Technology, and it was first introduced in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " 4.0."
}], "Text",
 CellID->1466712359],

Cell["An example shows a vector of real numbers.", "Text",
 CellID->69833800],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "10000", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"vec", "//", "ByteCount"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1170605338],

Cell[BoxData["80056"], "Output",
 ImageSize->{42, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1266225599]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"vec", "^", "2"}], ",", 
    RowBox[{"{", "50", "}"}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->2038184270],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.07999999999999999`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1251372593]
}, Open  ]],

Cell[TextData[{
 "Now insert a symbolic quantity into the vector. Updating elements of \
vectors and matrices is discussed under ",
 ButtonBox["Setting Pieces of a Matrix",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#1626148827"],
 "."
}], "Text",
 CellID->1511815480],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"vec", "[", 
   RowBox[{"[", "1", "]"}], "]"}], "=", "x"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->56261056],

Cell[BoxData["x"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->686763512]
}, Open  ]],

Cell["\<\
When the byte count is measured the vector is shown to use more memory.\
\>", "Text",
 CellID->1857227206],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"vec", "//", "ByteCount"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->742842097],

Cell[BoxData["200016"], "Output",
 ImageSize->{49, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1624684268]
}, Open  ]],

Cell["\<\
In addition, a mathematical operation on the vector is much slower.\
\>", "Text",
 CellID->813920671],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Do", "[", 
   RowBox[{
    RowBox[{"vec", "^", "2"}], ",", 
    RowBox[{"{", "50", "}"}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->93886107],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.561`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->1690803451]
}, Open  ]],

Cell["\<\
The first vector uses less memory and supports faster computations because it \
is built from a packed array. Packed arrays are a representation for \
rectangular tensors of machine integers, machine reals, and complex numbers \
with real and imaginary parts that are machine reals.\
\>", "Text",
 CellID->1829983853],

Cell[BoxData[GridBox[{
   {" ", Cell["machine\[Hyphen]sized integers", "TableText"]},
   {" ", Cell["machine\[Hyphen]precision reals", "TableText"]},
   {" ", Cell["\<\
machine\[Hyphen]precision complex numbers\
\>", "TableText"]}
  },
  GridBoxAlignment->{
   "Columns" -> {{Center}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
    "RowsIndexed" -> {}}]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}},
 GridBoxItemSize->{
  "Columns" -> {Left, {Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{"Columns" -> {
     Offset[0.28], {
      Offset[0.56]}, 
     Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
     Offset[0.2], {
      Offset[0.04]}, 
     Offset[0.2]}, "RowsIndexed" -> {}}},
 CellTags->"S3.1.1",
 CellID->277825037],

Cell["Packed array types. ", "Caption",
 CellTags->"S2.14.3",
 CellID->775948281],

Cell[TextData[{
 "There are a number of functions that are useful for working with packed \
arrays. ",
 StyleBox["Developer`PackedArrayQ", "MR"],
 " tests if its argument is a packed array. This demonstrates that the result \
of the ",
 Cell[BoxData[
  ButtonBox["Table",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Table"]], "InlineFormula"],
 " computation is a packed array."
}], "Text",
 CellID->1842054183],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "1000", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", "vec", "]"}]}], "Input",
 CellLabel->"In[7]:=",
 CellID->1280398529],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->1871933963]
}, Open  ]],

Cell["\<\
If a real number is inserted into the array, the result is still packed.\
\>", "Text",
 CellID->958687591],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"vec", "[", 
    RowBox[{"[", "2", "]"}], "]"}], "=", "10.1"}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", "vec", "]"}]}], "Input",
 CellLabel->"In[9]:=",
 CellID->92538496],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->516341844]
}, Open  ]],

Cell["\<\
If a computation is done with the packed array, the result is packed.\
\>", "Text",
 CellID->175954581],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec1", "=", 
   RowBox[{"vec", "^", "2"}]}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", "vec1", "]"}]}], "Input",
 CellLabel->"In[11]:=",
 CellID->183660948],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->855552297]
}, Open  ]],

Cell["\<\
However, if a general symbolic quantity is inserted into the packed array, \
the result is not packed.\
\>", "Text",
 CellID->1864439014],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"vec", "[", 
    RowBox[{"[", "2", "]"}], "]"}], "=", "x"}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", "vec", "]"}]}], "Input",
 CellLabel->"In[13]:=",
 CellID->1692614921],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->63105438]
}, Open  ]],

Cell["\<\
All the elements of the packed array must be of the same type. Therefore, if \
the integer 0 is inserted into a packed array of reals, the result is not \
packed.\
\>", "Text",
 CellID->1903658495],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "1000", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"vec", "[", 
    RowBox[{"[", "2", "]"}], "]"}], "=", "0"}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", "vec", "]"}]}], "Input",
 CellLabel->"In[15]:=",
 CellID->1576952459],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->1744290786]
}, Open  ]],

Cell[CellGroupData[{

Cell["Packed Array Functions", "Subsection",
 CellTags->{"t:177", "c:133"},
 CellID->995975105],

Cell["\<\
A number of functions for working with packed arrays are summarized below.\
\>", "Text",
 CellID->1866045017],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     StyleBox["Developer`PackedArrayQ", "MR"], "[", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "return ",
     ButtonBox["True",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/True"],
     " if ",
     StyleBox["expr", "TI"],
     " is a packed array"
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["Developer`ToPackedArray", "MR"], "[", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "convert ",
     StyleBox["expr", "TI"],
     " to a packed array if possible "
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["Developer`ToPackedArray", "MR"], "[", 
     StyleBox["expr", "TI"], ",", 
     StyleBox["t", "TI"], "]"}], Cell[TextData[{
     "convert ",
     StyleBox["expr", "TI"],
     " to a packed array of type ",
     StyleBox["t", "TI"],
     " if possible "
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["Developer`FromPackedArray", "MR"], "[", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "convert ",
     StyleBox["expr", "TI"],
     " from a packed array "
    }], "TableText"]},
   {
    RowBox[{
     StyleBox["Developer`PackedArrayForm", "MR"], "[", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "print packed arrays in ",
     StyleBox["expr", "TI"],
     " in a summary form "
    }], "TableText"]}
  }]], "DefinitionBox",
 CellTags->"S2.14.3",
 CellID->767419931],

Cell["Packed array functions.", "Caption",
 CellTags->"S2.14.3",
 CellID->239996435],

Cell[TextData[{
 "The function ",
 StyleBox["Developer`ToPackedArray", "MR"],
 " can be used to make a packed array."
}], "Text",
 CellID->645793979],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pack", "=", 
   RowBox[{"Developer`ToPackedArray", "[", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", "pack", "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->477053115],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1275249554]
}, Open  ]],

Cell["\<\
If the elements of the list are not all of the same type the result will not \
be a packed array.\
\>", "Text",
 CellID->1232457018],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pack", "=", 
   RowBox[{"Developer`ToPackedArray", "[", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3.5"}], "}"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", "pack", "]"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->1425326211],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->277545992]
}, Open  ]],

Cell[TextData[{
 "By specifying a type of ",
 Cell[BoxData[
  ButtonBox["Real",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Real"]], "InlineFormula"],
 ", the integers are coerced to reals and the result is packed."
}], "Text",
 CellID->381006260],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pack", "=", 
   RowBox[{"Developer`ToPackedArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3.5"}], "}"}], ",", "Real"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", "pack", "]"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->1923460113],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->1522780900]
}, Open  ]],

Cell[TextData[{
 "This shows that ",
 StyleBox["pack", "MR"],
 " is a packed array of three reals."
}], "Text",
 CellID->1815965001],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Developer`PackedArrayForm", "[", "pack", "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->886180643],

Cell[BoxData[
 RowBox[{"\<\"PackedArray\"\>", "[", 
  RowBox[{"Real", ",", 
   RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "3", 
    "\[InvisibleSpace]", "\<\">\"\>"}]}], "]"}]], "Output",
 ImageSize->{156, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1238974956]
}, Open  ]],

Cell["\<\
This does not make a packed array because the argument is not a rectangular \
tensor.\
\>", "Text",
 CellID->1450644180],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pack", "=", 
   RowBox[{"Developer`ToPackedArray", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "2"}], "}"}], ",", "2"}], "}"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", "pack", "]"}]}], "Input",
 CellLabel->"In[8]:=",
 CellID->594888869],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->1760844498]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Packed Array Operations", "Subsection",
 CellTags->{"t:182", "c:134"},
 CellID->41338579],

Cell[TextData[{
 "The important detail of packed arrays is that they work exactly like lists \
except for the packed array specific functions described ",
 ButtonBox["above",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraPerformance#995975105"],
 ". For example, ",
 Cell[BoxData[
  ButtonBox["SameQ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SameQ"]], "InlineFormula"],
 " on the packed and non-packed version of a list will return ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1140279272],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pack", "=", 
   RowBox[{"Developer`ToPackedArray", "[", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}], "]"}]}], ";"}], "\n", 
 RowBox[{"pack", "===", 
  RowBox[{"{", 
   RowBox[{"1", ",", "2", ",", "3"}], "}"}]}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->529658676],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->653650717]
}, Open  ]],

Cell["\<\
The packed and non-packed versions will behave the same in the pattern \
matcher.\
\>", "Text",
 CellID->1528746214],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatchQ", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", "_List"}], "]"}]], "Input",\

 CellLabel->"In[3]:=",
 CellID->1143265990],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1335304221]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"MatchQ", "[", 
  RowBox[{"pack", ",", "_List"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1253931197],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1564599968]
}, Open  ]],

Cell["\<\
Many functions can work more efficiently with packed arrays compared with \
unpacked lists, but the results will be the same, whether a packed or an \
unpacked version is used. In order to maintain consistency, the system must, \
on occasion, unpack a packed array. Because this is a potential source of \
inefficiency, a message is available to let you know that something has \
become unpacked. You can enable this message with a system option.\
\>", "Text",
 CellID->294404661],

Cell["\<\
This sets the system option and creates a packed array to use as an example.\
\>", "Text",
 CellID->256814739],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"Developer`SetSystemOptions", "[", " ", 
   RowBox[{"\"\<UnpackMessage\>\"", "\[Rule]", "True"}], "]"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"v", "=", 
   RowBox[{"Range", "[", "100", "]"}]}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", "v", "]"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->1990671550],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1569004237]
}, Open  ]],

Cell["\<\
Now if a real number is inserted into the packed array, it must be unpacked \
and a message is produced. This message can warn you that your code may not \
run as efficiently.\
\>", "Text",
 CellID->1302509746],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"v", "[", 
    RowBox[{"[", "1", "]"}], "]"}], "=", "5.6"}], ";"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->1980487151],

Cell[BoxData[
 RowBox[{
  RowBox[{"Developer`FromPackedArray", "::", "\<\"unpack1\"\>"}], ":", 
  " ", "\<\"Unpacking array. \\!\\(\\*ButtonBox[\\\"More\[Ellipsis]\\\", \
ButtonStyle->\\\"RefGuideNotebookLink\\\", ButtonFrame->None, \
ButtonData:>\\\"General::unpack1\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"From In[8]:=",
 CellID->51018357]
}, Open  ]],

Cell["It is usually a good idea to turn the message off again.", "Text",
 CellID->619550946],

Cell[BoxData[
 RowBox[{
  RowBox[{"Developer`SetSystemOptions", "[", " ", 
   RowBox[{"\"\<UnpackMessage\>\"", "\[Rule]", "False"}], "]"}], 
  ";"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->1792028766]
}, Open  ]],

Cell[CellGroupData[{

Cell["Packed Array Summary", "Subsection",
 CellTags->{"t:183", "c:135"},
 CellID->169320987],

Cell[TextData[{
 "The main point about packed arrays is that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " uses them automatically where appropriate. This is very much the case for \
the linear algebra functions, which naturally work with the same internal \
representation. Thus many functions return packed arrays even if the input is \
not packed. For example, the function ",
 Cell[BoxData[
  ButtonBox["Dot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Dot"]], "InlineFormula"],
 " returns a packed array because internally it works with the packed \
version."
}], "Text",
 CellID->361996234],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1.", ",", " ", "2."}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4.", ",", " ", "5."}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{"Developer`PackedArrayQ", "[", " ", 
  RowBox[{"mat", ".", "mat"}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->249532887],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->2011152865]
}, Open  ]],

Cell[TextData[{
 "To make use of packed arrays, it is typically hard to exploit functions \
such as ",
 StyleBox["Developer`ToPackedArray", "MR"],
 ". The main issue is to make sure that you have a uniformity of elements in \
your matrices; for example, that you are not mixing integers and reals."
}], "Text",
 CellID->53598275]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Programming Efficiency", "Section",
 CellTags->{"t:108", "c:136"},
 CellID->1030559832],

Cell[TextData[{
 "This section discusses techniques for writing efficient ",
 StyleBox["Mathematica ",
  FontSlant->"Italic"],
 "programs that solve linear algebra problems."
}], "Text",
 CellID->1163483369],

Cell[CellGroupData[{

Cell["Measuring Performance", "Subsection",
 CellTags->{"t:161", "c:137"},
 CellID->1826376848],

Cell[TextData[{
 "If you want to study the performance of your ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " code there are a number of timing related functions you can use."
}], "Text",
 CellID->1931917765],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["Timing",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Timing"], "[", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "evaluate ",
     StyleBox["expr", "TI"],
     " and return a list of the CPU time needed, together with the result \
obtained"
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["AbsoluteTiming",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/AbsoluteTiming"], "[", 
     StyleBox["expr", "TI"], "]"}], Cell[TextData[{
     "evaluate ",
     StyleBox["expr", "TI"],
     " giving the absolute time taken"
    }], "TableText"]}
  }]], "DefinitionBox",
 CellTags->"S2.14.3",
 CellID->1936617362],

Cell[TextData[{
 "Timing ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " operations. "
}], "Caption",
 CellTags->"S2.14.3",
 CellID->1465849202],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  ButtonBox["Timing",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Timing"]], "InlineFormula"],
 " function is particularly useful ",
 "because",
 " it returns the CPU time needed for the computation."
}], "Text",
 CellID->595139630],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "1000", "}"}], ",", 
     RowBox[{"{", "1000", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Timing", "[", 
  RowBox[{
   RowBox[{"mat", ".", "mat"}], ";"}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1680417331],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"1.673`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1216640420]
}, Open  ]],

Cell[TextData[{
 "Another useful function is ",
 Cell[BoxData[
  ButtonBox["TimeConstrained",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/TimeConstrained"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1883735853],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["TimeConstrained",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/TimeConstrained"], "[", 
     StyleBox["expr", "TI"], ",", 
     StyleBox["t", "TI"], "]"}], Cell[TextData[{
     "try to evaluate ",
     StyleBox["expr", "TI"],
     ", aborting the calculation after ",
     StyleBox["t", "TI"],
     " seconds "
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["TimeConstrained",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/TimeConstrained"], "[", 
     StyleBox["expr", "TI"], ",", 
     StyleBox["t", "TI"], ",", 
     StyleBox["failexpr", "TI"], "]"}], Cell[TextData[{
     "return ",
     StyleBox["failexpr", "TI"],
     " if the time constraint is not met "
    }], "TableText"]}
  }]], "DefinitionBox",
 CellTags->"S2.14.3",
 CellID->1744557297],

Cell["Time\[Hyphen]constrained calculation. ", "Caption",
 CellTags->"S2.14.3",
 CellID->1180258188],

Cell["\<\
This helps you to develop and test your code, shutting it down if it takes \
more than a preset limit.\
\>", "Text",
 CellID->292918377],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "1000", "}"}], ",", 
     RowBox[{"{", "1000", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"TimeConstrained", "[", " ", 
  RowBox[{
   RowBox[{"Do", "[", 
    RowBox[{
     RowBox[{"mat", ".", "mat"}], ",", 
     RowBox[{"{", "10", "}"}]}], "]"}], ",", "1"}], "]"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->1046643315],

Cell[BoxData["$Aborted"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->895642628]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Vectorizing Loops", "Subsection",
 CellTags->{"t:162", "c:138"},
 CellID->1352766973],

Cell["\<\
A common operation involves some type of iteration over the elements of a \
matrix. Because the matrices may be large it is obviously important to do \
this efficiently.\
\>", "Text",
 CellID->545144422],

Cell[TextData[{
 "One of the most important ways to write efficient code is to avoid explicit \
part references, especially in inner loops. This is a major source of \
inefficiency in user code and can be avoided by using functions that work on \
all the elements of a matrix or a vector at once. These vectorized operations \
allow ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " to use highly optimized code that runs faster."
}], "Text",
 CellID->1077062452],

Cell[CellGroupData[{

Cell["List Creation", "Subsubsection",
 CellTags->{"t:163", "c:139"},
 CellID->1048970201],

Cell["\<\
This problem involves creating a list and then initializing it with certain \
values. For example, computing \
\>", "Text",
 CellID->1348909984],

Cell["\t\tv\[LeftDoubleBracket]i\[RightDoubleBracket] \[DoubleLongRightArrow] \
Sin[i]", "Program",
 CellID->145239709],

Cell[TextData[{
 "One way to do this would be to create the list and then to use a ",
 Cell[BoxData[
  ButtonBox["For",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/For"]], "InlineFormula"],
 " loop that iterates over its elements to set them to a value."
}], "Text",
 CellID->1502366171],

Cell["\<\
\tn = 10;\t
\tv = Table[0, {n}];
\tFor[i = 1, i <= n, i++, v[[i]] = Sin[2.0 Pi i]];\
\>", "Program",
 CellID->155514571],

Cell["Slow way to initialize a list. ", "Caption",
 CellTags->"S3.7.2",
 CellID->245796398],

Cell["\<\
It is much faster to create the list and initialize it all at once; it is \
also neater. \
\>", "Text",
 CellID->1489008005],

Cell["\<\
\tn = 10;
\tv = Table[ Sin[2.0 Pi i], {i, 1, n}]\
\>", "Program",
 CellID->1130601544],

Cell["Faster way to initialize a list. ", "Caption",
 CellTags->"S3.7.2",
 CellID->1084841352],

Cell[TextData[{
 "An even faster way is to use the optimized function ",
 Cell[BoxData[
  ButtonBox["Range",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Range"]], "InlineFormula"],
 " to create the list and then to use vectorized operations."
}], "Text",
 CellID->792440034],

Cell["\<\
\tn = 10;
\tv = Sin[2.0 Pi Range[1., n]]\
\>", "Program",
 CellID->1993556391],

Cell["An even faster way to initialize a list. ", "Caption",
 CellTags->"S3.7.2",
 CellID->662403043],

Cell[TextData[{
 "Commands such as ",
 Cell[BoxData[
  ButtonBox["Table",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Table"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Range",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Range"]], "InlineFormula"],
 " are very optimized for building lists and matrices. They are discussed \
under \"",
 ButtonBox["Building Matrices",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#441715480"],
 "\"."
}], "Text",
 CellID->452717653]
}, Open  ]],

Cell[CellGroupData[{

Cell["List Updating", "Subsubsection",
 CellTags->{"t:164", "c:140"},
 CellID->291032650],

Cell[TextData[{
 "This problem involves updating the elements of a list; for example, \
computing the ",
 Cell[BoxData[
  ButtonBox["Sin",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sin"]], "InlineFormula"],
 " of every element."
}], "Text",
 CellID->1565484331],

Cell["\t\tv\[LeftDoubleBracket]i\[RightDoubleBracket] \[DoubleLongRightArrow] \
Sin[v\[LeftDoubleBracket]i\[RightDoubleBracket]]", "Program",
 CellID->125508073],

Cell["\<\
One way to do this involves iterating over the elements of the list replacing \
each with the updated version.\
\>", "Text",
 CellID->585241557],

Cell["\tDo[v[[i]] = Sin[v[[i]]], {i, n}];", "Program",
 CellID->679745239],

Cell["Slow way to update a list. ", "Caption",
 CellTags->"S3.7.2",
 CellID->2012713480],

Cell["\<\
It is much faster to compute the new values by using a vectorized operation. \
The code is also neater and tidier.\
\>", "Text",
 CellID->513050816],

Cell["\tv = Sin[v]", "Program",
 CellID->698221807],

Cell["Faster way to update a list. ", "Caption",
 CellTags->"S3.7.2",
 CellID->1163263041],

Cell["\<\
Another form of updating requires another argument that is not constant \
across the list.\
\>", "Text",
 CellID->270807799],

Cell[TextData[{
 "\t\tv\[LeftDoubleBracket]i\[RightDoubleBracket] \[DoubleLongRightArrow] ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"v", "\[LeftDoubleBracket]", "i", "\[RightDoubleBracket]"}], " ", 
    "/", 
    SuperscriptBox["i", "2"]}], TraditionalForm]]]
}], "Program",
 CellID->948829800],

Cell["\<\
One way to do this involves iterating over the elements of the list, dividing \
each by the square of the index.\
\>", "Text",
 CellID->1916369625],

Cell["\<\
\tn = Length[v];
\tDo[ v[[i]] = v[[i]]/i^2 , {i, n}];\
\>", "Program",
 CellID->464549669],

Cell["Slow way to update a list. ", "Caption",
 CellTags->"S3.7.2",
 CellID->1583230038],

Cell["\<\
It is faster to compute the list of updates and then carry out a vectorized \
division.\
\>", "Text",
 CellID->2024862336],

Cell[BoxData[
 RowBox[{"\t", 
  RowBox[{
   RowBox[{
    RowBox[{"d", "=", 
     SuperscriptBox[
      RowBox[{"Range", "[", "n", "]"}], "2"]}], ";"}], "\[IndentingNewLine]", 
   "\t", 
   RowBox[{"v", "=", 
    RowBox[{"v", "/", "d"}]}]}]}]], "Program",
 CellID->687927143],

Cell["Faster way to update a list. ", "Caption",
 CellTags->"S3.7.2",
 CellID->2087020600],

Cell[TextData[{
 "The use of listable operations is discussed under \"",
 ButtonBox["Listability",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#540401315"],
 "\"."
}], "Text",
 CellID->1704817749]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Using Built-in Support", "Subsection",
 CellTags->{"t:165", "c:141"},
 CellID->1145089169],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " has a very wide range of functions, including many for list processing \
that have important applications for writing matrix computations. Before you \
start to work it is worth looking to see if any of the built-in functions \
will do what you want. For example, if what you are doing involves a \
convolution or correlation then you should use the appropriate functions; \
they will almost certainly be faster."
}], "Text",
 CellID->1912702616],

Cell[TextData[{
 "Another important point to remember is that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can represent a wide range of mathematical objects. For example, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can work with the actual linear equations that matrices are used to \
represent. Sometimes the equational form can be more expressive and \
convenient and this can lead to greater efficiency. The topic is discussed \
under \"",
 ButtonBox["Converting Equations to Sparse Arrays",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraSparseArrays#139087150"],
 "\"."
}], "Text",
 CellID->1110155169],

Cell[TextData[{
 "One area of built-in functions with many applications for linear algebra \
computations are those for structural manipulation of lists. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " functions such as ",
 Cell[BoxData[
  ButtonBox["Part",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Part"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Take",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Take"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Drop",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Drop"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["Transpose",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Transpose"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["PadLeft",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PadLeft"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["PadRight",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PadRight"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["RotateLeft",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotateLeft"]], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  ButtonBox["RotateRight",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotateRight"]], "InlineFormula"],
 " are all heavily optimized and have many applications. These are discussed \
in more detail under \"",
 ButtonBox["Structural Operations",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#223720584"],
 "\". It is usually good if you can write your computations to use these \
functions."
}], "Text",
 CellID->1040000247],

Cell["\<\
A worked example will now be given that discusses different techniques to \
extend a matrix by tiling copies. For example, the input matrix\
\>", "Text",
 CellID->914748396],

Cell[TextData[{
 "\t\t\t",
 Cell[BoxData[
  RowBox[{"(", GridBox[{
     {"1", "2", "3"},
     {"4", "5", "6"}
    }], ")"}]]]
}], "Program",
 CellID->340862699],

Cell["should be padded out to return the following matrix.", "Text",
 CellID->1540495544],

Cell[TextData[{
 "\t\t\t",
 Cell[BoxData[
  RowBox[{"(", GridBox[{
     {"1", "2", "3", "1", "2", "3", "1", "2", "3"},
     {"4", "5", "6", "4", "5", "6", "4", "5", "6"},
     {"1", "2", "3", "1", "2", "3", "1", "2", "3"},
     {"4", "5", "6", "4", "5", "6", "4", "5", "6"}
    }], ")"}]]]
}], "Program",
 CellID->1045759776],

Cell["\<\
Three different ways to solve the problem will be demonstrated.\
\>", "Text",
 CellID->1078388184],

Cell[CellGroupData[{

Cell["A Slow Way", "Subsubsection",
 CellTags->{"t:166", "c:142"},
 CellID->1855711455],

Cell["\<\
The slowest way to do this is to write this in a step by step approach \
directly manipulating every element.\
\>", "Text",
 CellID->1037418804],

Cell[BoxData[
 RowBox[{
  RowBox[{"extendMatSlow", "[", 
   RowBox[{"mat_", ",", "nRow_", ",", "nCol_"}], "]"}], ":=", 
  "\[IndentingNewLine]", 
  RowBox[{"Module", "[", " ", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"m", ",", "n", ",", "mf", ",", "nf", ",", "final"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"m", ",", "n"}], "}"}], "=", 
      RowBox[{"Dimensions", "[", "mat", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"mf", " ", "=", " ", 
      RowBox[{"m", "*", "nRow"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"nf", "=", " ", 
      RowBox[{"n", "*", "nCol"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"final", "=", 
      RowBox[{"Table", "[", 
       RowBox[{"0", ",", 
        RowBox[{"{", "mf", "}"}], ",", 
        RowBox[{"{", "nf", "}"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"Do", "[", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"Do", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"final", "[", 
           RowBox[{"[", 
            RowBox[{
             RowBox[{
              RowBox[{"m", "*", 
               RowBox[{"(", 
                RowBox[{"i", "-", "1"}], ")"}]}], "+", "i1"}], ",", 
             RowBox[{
              RowBox[{"n", "*", 
               RowBox[{"(", 
                RowBox[{"j", "-", "1"}], ")"}]}], "+", "j1"}]}], "]"}], "]"}],
           "=", 
          RowBox[{"mat", "[", 
           RowBox[{"[", 
            RowBox[{"i1", ",", "j1"}], "]"}], "]"}]}], ",", 
         RowBox[{"{", 
          RowBox[{"i1", ",", "m"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"j1", ",", "n"}], "}"}]}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "nRow"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"j", ",", "nCol"}], "}"}]}], "]"}], ";", 
     "\[IndentingNewLine]", "final"}]}], "\[IndentingNewLine]", 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->478108835],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"extendMatSlow", "[", 
   RowBox[{"mat", ",", "2", ",", "3"}], "]"}], "//", "MatrixForm"}]}], "Input",\

 CellLabel->"In[2]:=",
 CellID->1838866418],

Cell[BoxData[
 TagBox[
  RowBox[{"(", "\[NoBreak]", GridBox[{
     {"1", "2", "3", "1", "2", "3", "1", "2", "3"},
     {"4", "5", "6", "4", "5", "6", "4", "5", "6"},
     {"1", "2", "3", "1", "2", "3", "1", "2", "3"},
     {"4", "5", "6", "4", "5", "6", "4", "5", "6"}
    }], "\[NoBreak]", ")"}],
  Function[BoxForm`e$, 
   MatrixForm[BoxForm`e$]]]], "Output",
 ImageSize->{151, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]//MatrixForm=",
 CellID->1390723349]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "100", "}"}], ",", 
     RowBox[{"{", "20", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"extendMatSlow", "[", 
    RowBox[{"mat", ",", "10", ",", "15"}], "]"}], ";"}], "//", 
  "Timing"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->675312695],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"2.253`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->529390470]
}, Open  ]],

Cell[TextData[{
 "This way suffers from several speed deficiencies, the most severe being the \
inner loops over the part indices. These would be much faster if they were \
replaced by vectorized code. It was explained ",
 ButtonBox["earlier",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraPerformance#1826376848"],
 " that explicit part references in an inner loop can lead to inefficient \
code."
}], "Text",
 CellID->440879769],

Cell["\<\
One thing to note about the function is that it will work when given a sparse \
matrix as input, but it will return a dense matrix as output.\
\>", "Text",
 CellID->1047695433],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sp", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], "\[Rule]", "4"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "3"}], "}"}], "\[Rule]", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"extendMatSlow", "[", 
  RowBox[{"sp", ",", "2", ",", "3"}], "]"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->1398720678],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "0", ",", "4", ",", "0", ",", "0", ",", "4", ",", "0", ",", "0", ",", "4",
      ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0",
      ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "4", ",", "0", ",", "0", ",", "4", ",", "0", ",", "0", ",", "4",
      ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0",
      ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1942435084]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["A Faster Way", "Subsubsection",
 CellTags->{"t:167", "c:143"},
 CellID->460055094],

Cell[TextData[{
 "A much faster way to replicate the matrix is to write it to use ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " vectorizing functions."
}], "Text",
 CellID->1204835658],

Cell[BoxData[
 RowBox[{
  RowBox[{"extendMatFaster", "[", 
   RowBox[{"mat_", ",", "nRow_", ",", "nCol_"}], "]"}], ":=", 
  "\[IndentingNewLine]", 
  RowBox[{"Module", "[", " ", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"m", ",", "n", ",", "mind", ",", "nind"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"m", ",", "n"}], "}"}], "=", 
      RowBox[{"Dimensions", "[", "mat", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"mind", "=", 
      RowBox[{"Range", "[", "m", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"mind", "=", 
      RowBox[{"Flatten", "[", 
       RowBox[{"Table", "[", 
        RowBox[{"mind", ",", 
         RowBox[{"{", "nRow", "}"}]}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"nind", "=", 
      RowBox[{"Range", "[", "n", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"nind", "=", 
      RowBox[{"Flatten", "[", 
       RowBox[{"Table", "[", 
        RowBox[{"nind", ",", 
         RowBox[{"{", "nCol", "}"}]}], "]"}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"mat", "[", 
      RowBox[{"[", 
       RowBox[{"mind", ",", "nind"}], "]"}], "]"}]}]}], "\[IndentingNewLine]",
    "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1049596443],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{"extendMatFaster", "[", 
  RowBox[{"mat", ",", "2", ",", "4"}], "]"}]}], "Input",
 CellLabel->"In[2]:=",
 CellID->1136741043],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "1", ",", "2", ",", "3", ",", "1", ",", "2",
      ",", "3", ",", "1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "4", ",", "5", ",", "6", ",", "4", ",", "5", ",", "6", ",", "4", ",", "5",
      ",", "6", ",", "4", ",", "5", ",", "6"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "1", ",", "2", ",", "3", ",", "1", ",", "2",
      ",", "3", ",", "1", ",", "2", ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "4", ",", "5", ",", "6", ",", "4", ",", "5", ",", "6", ",", "4", ",", "5",
      ",", "6", ",", "4", ",", "5", ",", "6"}], "}"}]}], "}"}]], "Output",
 ImageSize->{472, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1192185989]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"Random", ",", 
     RowBox[{"{", "100", "}"}], ",", 
     RowBox[{"{", "20", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"extendMatFaster", "[", 
    RowBox[{"mat", ",", "10", ",", "15"}], "]"}], ";"}], "//", 
  "Timing"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->1047490613],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.05999999999999872`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->103794585]
}, Open  ]],

Cell[TextData[{
 "This method is much faster; it uses the function ",
 Cell[BoxData[
  ButtonBox["Part",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Part"]], "InlineFormula"],
 ", which is very flexible and heavily optimized. It is described further \
under ",
 ButtonBox["Getting Pieces of a Matrix",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#1678167046"],
 "."
}], "Text",
 CellID->718868731],

Cell["\<\
This function works for a sparse matrix as input, and it returns the result \
as a sparse array which maintains equivalent sparsity.\
\>", "Text",
 CellID->283763895],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sp", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], "\[Rule]", "4"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "3"}], "}"}], "\[Rule]", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"extendMatFaster", "[", 
  RowBox[{"sp", ",", "2", ",", "3"}], "]"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->1806086925],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "12", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 12, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "9"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->221598130]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Also Fast but Neater", "Subsubsection",
 CellTags->{"t:168", "c:144"},
 CellID->663127012],

Cell[TextData[{
 "Another way to extend the matrix, which involves no programming, is to use \
the built-in function ",
 Cell[BoxData[
  ButtonBox["PadRight",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PadRight"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->1883046401],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5", ",", "6"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{"PadRight", "[", 
  RowBox[{"mat", ",", 
   RowBox[{"{", 
    RowBox[{"4", ",", "9"}], "}"}], ",", "mat"}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->894355768],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "1", ",", "2", ",", "3", ",", "1", ",", "2",
      ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "4", ",", "5", ",", "6", ",", "4", ",", "5", ",", "6", ",", "4", ",", "5",
      ",", "6"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "1", ",", "2", ",", "3", ",", "1", ",", "2", ",", "3", ",", "1", ",", "2",
      ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
    "4", ",", "5", ",", "6", ",", "4", ",", "5", ",", "6", ",", "4", ",", "5",
      ",", "6"}], "}"}]}], "}"}]], "Output",
 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1199347423]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"Random", ",", 
     RowBox[{"{", "100", "}"}], ",", 
     RowBox[{"{", "20", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"PadRight", "[", 
    RowBox[{"mat", ",", 
     RowBox[{"{", 
      RowBox[{"1000", ",", "300"}], "}"}], ",", "mat"}], "]"}], ";"}], "//", 
  "Timing"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->1150627619],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.05000000000000071`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->508151298]
}, Open  ]],

Cell[TextData[{
 "This method is also neater and is probably the fastest for small input \
matrices. For large matrices, it is similar to the speed of the ",
 Cell[BoxData[
  ButtonBox["Part",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Part"]], "InlineFormula"],
 " example shown ",
 ButtonBox["above",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraPerformance#460055094"],
 ". The use of PadRight is described under \"",
 ButtonBox["Extending Matrices",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#782554218"],
 "\"."
}], "Text",
 CellID->2103885014],

Cell["\<\
This function works for a sparse matrix as input, and it returns the result \
as a sparse array which maintains equivalent sparsity.\
\>", "Text",
 CellID->1677182443],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sp", "=", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "2"}], "}"}], "\[Rule]", "4"}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"2", ",", "3"}], "}"}], "\[Rule]", "1"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"PadRight", "[", 
  RowBox[{"sp", ",", 
   RowBox[{"{", 
    RowBox[{"4", ",", "9"}], "}"}], ",", "sp"}], "]"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->534968979],

Cell[BoxData[
 TagBox[
  RowBox[{"SparseArray", "[", 
   RowBox[{
    InterpretationBox[
     RowBox[{"\<\"<\"\>", "\[InvisibleSpace]", "12", 
      "\[InvisibleSpace]", "\<\">\"\>"}],
     SequenceForm["<", 12, ">"],
     Editable->False], ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "9"}], "}"}]}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->1857083711]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Matrix Contents", "Subsection",
 CellTags->{"t:184", "c:145"},
 CellID->478605279],

Cell[TextData[{
 "Matrices in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can be constructed from all the different types of object that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " holds. They can contain machine-precision floating point numbers, \
arbitrary-precision floating point numbers, complex floating point numbers, \
integers, rational numbers, and general symbolic quantities. The different \
types are discussed under \"",
 ButtonBox["Matrix Types",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixTypes#1880892864"],
 "\"."
}], "Text",
 CellID->1592666138],

Cell[TextData[{
 "That ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can work with these different types of matrix is a great strength of the \
system. However, it has a number of implications for the efficiency of \
numerical floating point computation: first from a mix of symbolic and \
numerical entries, secondly from a mix of different types of numbers, and \
thirdly from integer matrices."
}], "Text",
 CellID->1703605158],

Cell[CellGroupData[{

Cell["Mixed Symbolic/Numerical Matrices", "Subsubsection",
 CellTags->{"t:178", "c:146"},
 CellID->92425055],

Cell[TextData[{
 "If you mix symbolic and numerical elements in a matrix, the linear algebra \
functions will treat the matrix as symbolic as discussed under \"",
 ButtonBox["Mixed Mode Matrices",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixTypes#1172699025"],
 "\". It should be noted that techniques for working with symbolic matrices \
are typically slower than the heavily optimized techniques available for \
numerical matrices."
}], "Text",
 CellID->757222604],

Cell["\<\
An example here compares matrix multiplication of two 200\[Cross]200 \
matrices. The second computation, which involves a symbolic matrix, is much \
slower.\
\>", "Text",
 CellID->652213306],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "200", "}"}], ",", 
     RowBox[{"{", "200", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"mat", ".", "mat"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->992499654],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.030000000000001137`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1436198529]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"mat", "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "2"}], "]"}], "]"}], "=", "x"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"mat", ".", "mat"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->472484623],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"3.295`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1802015802]
}, Open  ]],

Cell["\<\
The solution is to make sure that your matrices only contain numbers.\
\>", "Text",
 CellID->517710573]
}, Open  ]],

Cell[CellGroupData[{

Cell["Mixed Numerical Type Matrices", "Subsubsection",
 CellTags->{"t:185", "c:147"},
 CellID->1705346414],

Cell[TextData[{
 "If you mix different types of numbers in your matrices the linear algebra \
functions will treat the matrix as numerical and will coerce the matrix to \
the lowest precision. This is discussed under \"",
 ButtonBox["Mixed Mode Matrices",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixTypes#1172699025"],
 "\". For linear algebra functions this will not have as severe an impact on \
performance as when working with symbolic matrices (this was demonstrated in \
the ",
 ButtonBox["previous",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraPerformance#92425055"],
 " section)."
}], "Text",
 CellID->1440427236],

Cell["\<\
This example compares matrix multiplication of two 200\[Cross]200 matrices. \
The second matrix, which contains an integer, is slower for matrix/matrix \
multiplication. If the operation was something more costly, this difference \
might not be significant.\
\>", "Text",
 CellID->677402821],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "200", "}"}], ",", 
     RowBox[{"{", "200", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"mat", ".", "mat"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->1298399800],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.03`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1624101433]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"mat", "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "2"}], "]"}], "]"}], "=", "1"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"mat", ".", "mat"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->1755631061],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.04999999999999999`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->465932447]
}, Open  ]],

Cell[TextData[{
 "The cost of mixed numerical matrices comes because ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " cannot use its efficient storage techniques, as discussed in the section \
\"",
 ButtonBox["Packed Arrays",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraPerformance#1945433762"],
 "\". This is demonstrated in the following example. Here, a vector of \
numbers is allocated and this is updated by some process. Because the updated \
values will be real numbers and the initial vector contained integers, the \
process cannot use packed arrays."
}], "Text",
 CellID->1201259767],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "300"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"0", ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"ind", "=", 
       RowBox[{"Random", "[", 
        RowBox[{"Integer", ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "n"}], "}"}]}], "]"}]}], ";", 
      RowBox[{
       RowBox[{"vec", "[", 
        RowBox[{"[", "ind", "]"}], "]"}], "=", " ", 
       RowBox[{"Total", "[", 
        RowBox[{"1.5", "+", 
         RowBox[{"2", " ", "vec"}]}], "]"}]}]}], ",", 
     RowBox[{"{", "100", "}"}]}], "]"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->1079138522],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.09999999999999998`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{124, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1581740339]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Developer`PackedArrayQ", "[", "vec", "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->609811360],

Cell[BoxData["False"], "Output",
 ImageSize->{39, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->1307564036]
}, Open  ]],

Cell["\<\
In this version the vector is initialized with real numbers. Now the \
computation can use packed arrays throughout and it runs faster.\
\>", "Text",
 CellID->1028593301],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "300"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Table", "[", 
    RowBox[{"0.0", ",", 
     RowBox[{"{", "n", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"ind", "=", 
       RowBox[{"Random", "[", 
        RowBox[{"Integer", ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "n"}], "}"}]}], "]"}]}], ";", 
      RowBox[{
       RowBox[{"vec", "[", 
        RowBox[{"[", "ind", "]"}], "]"}], "=", 
       RowBox[{"Total", "[", 
        RowBox[{"1.5", "+", 
         RowBox[{"2", " ", "vec"}]}], "]"}]}]}], ",", 
     RowBox[{"{", "100", "}"}]}], "]"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[9]:=",
 CellID->324251557],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{117, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->1529906060]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Developer`PackedArrayQ", "[", "vec", "]"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->1309444889],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->427834967]
}, Open  ]],

Cell["\<\
The solution to this problem is to make sure you initialize with real numbers \
if you are going to work with real numbers.\
\>", "Text",
 CellID->847040213]
}, Open  ]],

Cell[CellGroupData[{

Cell["Integer Matrices", "Subsubsection",
 CellTags->{"t:186", "c:148"},
 CellID->535763928],

Cell[TextData[{
 "A final type of problem involves working with purely integer matrices. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " makes a distinction between working with integer and approximate real \
matrices; this is discussed under \"",
 ButtonBox["Matrices in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixTypes#77896732"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraMatrixTypes#77896732"],
  FontSlant->"Italic"],
 "\". Because an integer matrix will use symbolic computation techniques, it \
will be slower. Of course, if ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " did not use integer matrix techniques with matrices of integers it would \
not be so general."
}], "Text",
 CellID->185406025],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", " ", 
    RowBox[{
     RowBox[{"Eigenvalues", "[", "mat", "]"}], ",", 
     RowBox[{"{", "100", "}"}]}], "]"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->120290378],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.24099999999999966`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->938271219]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"mat", "=", 
   RowBox[{"N", "[", "mat", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", " ", 
    RowBox[{
     RowBox[{"Eigenvalues", "[", "mat", "]"}], ",", 
     RowBox[{"{", "100", "}"}]}], "]"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->551957812],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{117, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1773074891]
}, Open  ]],

Cell["\<\
The difference between the two computations can be compared. The eigenvalues \
of the integer matrix are returned with the exact values; in this case this \
involves radicals. The eigenvalues of the real matrix are returned with \
approximate real values. \
\>", "Text",
 CellID->1586607265],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Eigenvalues", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"3", ",", "4"}], "}"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->803858056],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"(", 
     RowBox[{"5", "+", 
      SqrtBox["33"]}], ")"}]}], ",", 
   RowBox[{
    FractionBox["1", "2"], " ", 
    RowBox[{"(", 
     RowBox[{"5", "-", 
      SqrtBox["33"]}], ")"}]}]}], "}"}]], "Output",
 ImageSize->{187, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->622723915]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Eigenvalues", "[", 
  RowBox[{"N", "[", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "4"}], "}"}]}], "}"}], "]"}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1433027574],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"5.372281323269014`", ",", 
   RowBox[{"-", "0.3722813232690143`"}]}], "}"}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->865335113]
}, Open  ]],

Cell[TextData[{
 "Of course, it is a strength of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " that it can return the exact result for a computation involving an integer \
matrix. But it should be noted that many purely numerical systems would \
return the approximate result."
}], "Text",
 CellID->927986707],

Cell["\<\
The solution is that to work with matrices of approximate real numbers, you \
should start with approximate real numbers.\
\>", "Text",
 CellID->1646340035]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Expression Efficiency", "Section",
 CellTags->{"t:108", "c:149"},
 CellID->1857660297],

Cell[TextData[{
 "\"",
 ButtonBox["Matrices as ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraIntroduction#1127257628"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraIntroduction#1127257628"],
  FontSlant->"Italic"],
 ButtonBox[" Expressions",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraIntroduction#1127257628"],
 "\" explained that ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " represents matrices as expressions, a uniform data type that is used \
throughout the system. It described the advantages that came from this \
uniformity in the system. There are a number of efficiency implications that \
arise from the use of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expressions. This section will discuss these."
}], "Text",
 CellID->1579351593],

Cell[CellGroupData[{

Cell["Updating of Matrices", "Subsection",
 CellTags->{"t:187", "c:150"},
 CellID->1264346631],

Cell[TextData[{
 "A general principle of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expressions is that they cannot be changed directly. This immutability of \
expressions is very useful; it allows an expression to be shared in many \
locations and each use need not be concerned that some other part of the code \
will modify the expression. This means that when an expression is updated \
(this was described under ",
 ButtonBox["Setting Pieces of a Matrix",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations#1626148827"],
 "), it may be necessary to make a copy. In this example, a vector is made \
and assigned to two different symbols."
}], "Text",
 CellID->523140152],

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec1", "=", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vec2", "=", "vec1"}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->618483846],

Cell["Now change the contents of one symbol.", "Text",
 CellID->1037148242],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"vec1", "[", 
    RowBox[{"[", "1", "]"}], "]"}], "=", "4"}], ";"}], "\n", "vec1"}], "Input",\

 CellLabel->"In[3]:=",
 CellID->225740595],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4", ",", "2", ",", "3"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1566059563]
}, Open  ]],

Cell["\<\
However, the other one is left unchanged. This is because the expression was \
copied before modification.\
\>", "Text",
 CellID->1203497709],

Cell[CellGroupData[{

Cell[BoxData["vec2"], "Input",
 CellLabel->"In[5]:=",
 CellID->37524070],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "2", ",", "3"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1454335919]
}, Open  ]],

Cell["\<\
This copying can have an implication for efficiency, especially if you are \
iterating over the elements of a matrix and updating it. If this is done and \
you also have other copies of the matrix, it will need to be copied. The \
principle is demonstrated in the following two examples. In this first \
example there are no extra copies of the vector and so it does not need to be \
copied and the loop runs faster.\
\>", "Text",
 CellID->364920479],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Range", "[", 
    RowBox[{"1.", ",", "50000"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"vec", "[", 
       RowBox[{"[", "i", "]"}], "]"}], "=", 
      RowBox[{"i", "-", "1"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1.", ",", "50000"}], "}"}]}], "]"}], ";"}], "//", 
  "Timing"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->5194721],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.20999999999999908`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1408329967]
}, Open  ]],

Cell["\<\
In this example there are extra copies of the vector and so it does need to \
be copied at each step; the loop runs more slowly.\
\>", "Text",
 CellID->236998135],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Range", "[", 
    RowBox[{"1.", ",", "50000"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"vec1", "=", "vec"}], ";", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"vec", "[", 
        RowBox[{"[", "i", "]"}], "]"}], "=", 
       RowBox[{"i", "-", "1"}]}]}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1.", ",", "50000"}], "}"}]}], "]"}], ";"}], "//", 
  "Timing"}]}], "Input",
 CellLabel->"In[8]:=",
 CellID->1792973407],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"6.468999999999998`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->1013516745]
}, Open  ]],

Cell[TextData[{
 "Another slow way to update elements is to use ",
 Cell[BoxData[
  ButtonBox["ReplacePart",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/ReplacePart"]], "InlineFormula"],
 ". This also has to copy at every step."
}], "Text",
 CellID->335645211],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"Range", "[", 
    RowBox[{"1.", ",", "50000"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", 
    RowBox[{
     RowBox[{"ReplacePart", "[", 
      RowBox[{"vec", ",", 
       RowBox[{
        RowBox[{"N", "[", "i", "]"}], "-", "1"}], ",", "i"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1", ",", "50000"}], "}"}]}], "]"}], ";"}], "//", 
  "Timing"}]}], "Input",
 CellLabel->"In[10]:=",
 CellID->1897004255],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"6.420000000000002`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->1408290423]
}, Open  ]],

Cell[TextData[{
 "If you use the vectorizing operations described ",
 ButtonBox["previously",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraPerformance#1352766973"],
 ", this is a good way to avoid the iteration over the matrix and minimize \
the number of copying operations. If you really cannot avoid iterating over \
the matrix, it is a good idea to keep the loop as simple and tidy as possible \
so that you can avoid extra copies of the matrix."
}], "Text",
 CellID->1719865774],

Cell["\<\
The time for the overall process will scale with the square of the length of \
the list for the technique that has to copy at every step. With no copying \
the time will scale linearly. The longer the input list, the greater the \
impact, and the performance of the code will start to degrade.\
\>", "Text",
 CellID->654174998]
}, Open  ]],

Cell[CellGroupData[{

Cell["Appending to Matrices", "Subsection",
 CellTags->{"t:188", "c:151"},
 CellID->1750402393],

Cell[TextData[{
 "In ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", expressions are implemented as arrays. This allows fast access to any \
particular elements of an expression. Because matrices are regular ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " expressions, they are also arrays. While arrays are very fast for \
accessing elements, they are slower for adding elements. Typically this \
requires a copy, and it should be avoided. The example that follows shows the \
cost of adding elements."
}], "Text",
 CellID->634538898],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"vec", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Do", "[", 
    RowBox[{
     RowBox[{"AppendTo", "[", 
      RowBox[{"vec", ",", "i"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "10000"}], "}"}]}], "]"}], ";"}], "//", 
  "Timing"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->291467684],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.7310000000000016`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->287166181]
}, Open  ]],

Cell["\<\
It is much more efficient to generate the entire vector at once.\
\>", "Text",
 CellID->1006249258],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"vec", "=", 
    RowBox[{"Table", "[", 
     RowBox[{"i", ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "10000"}], "}"}]}], "]"}]}], ";"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->597559081],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.`", " ", "Second"}], ",", "Null"}], "}"}]], "Output",
 ImageSize->{117, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1541940995]
}, Open  ]],

Cell["\<\
If the arguments are not known all at once, other techniques may be used. \
These are demonstrated below. First, build a list of random numbers.\
\>", "Text",
 CellID->1717134616],

Cell[BoxData[{
 RowBox[{
  RowBox[{"num", "=", "10000"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"list", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Random", "[", "]"}], ",", 
     RowBox[{"{", "num", "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->1996242726],

Cell[TextData[{
 "The first technique is using ",
 Cell[BoxData[
  ButtonBox["AppendTo",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AppendTo"]], "InlineFormula"],
 "."
}], "Text",
 CellID->714478801],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"res1", " ", "=", " ", 
   RowBox[{"{", "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"#", ">", "0.5"}], ",", 
        RowBox[{"AppendTo", "[", 
         RowBox[{"res1", ",", "#"}], "]"}]}], "]"}], "&"}], ",", "list"}], 
    "]"}], ";"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->1664138030],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.21000000000000085`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->340851682]
}, Open  ]],

Cell[TextData[{
 "It is much faster to accumulate the list, inserting empty lists that can be \
removed at the end with ",
 Cell[BoxData[
  ButtonBox["Flatten",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Flatten"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1355332825],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"res2", "=", 
    RowBox[{"Flatten", "[", 
     RowBox[{"Map", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"If", "[", 
         RowBox[{
          RowBox[{"#", ">", "0.5"}], ",", "#", ",", 
          RowBox[{"{", "}"}]}], "]"}], "&"}], ",", "list"}], "]"}], "]"}]}], 
   ";"}], "//", "Timing"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->719746046],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.030000000000001137`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->2143895218]
}, Open  ]],

Cell[TextData[{
 "It may be inconvenient to use ",
 Cell[BoxData[
  ButtonBox["Flatten",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Flatten"]], "InlineFormula"],
 " because this might interfere with the structure of the list. In this case \
it might be convenient to use ",
 Cell[BoxData[
  ButtonBox["Sow",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Sow"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Reap",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reap"]], "InlineFormula"],
 ". This is more flexible than the ",
 Cell[BoxData[
  ButtonBox["Flatten",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Flatten"]], "InlineFormula"],
 " approach."
}], "Text",
 CellID->1966299560],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"res3", " ", "=", " ", 
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"Map", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"If", "[", 
          RowBox[{
           RowBox[{"#", ">", "0.5"}], ",", 
           RowBox[{"Sow", "[", "#", "]"}]}], "]"}], "&"}], ",", "list"}], 
       "]"}], "]"}], "[", 
     RowBox[{"[", 
      RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->2118344884],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.029999999999997584`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{131, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->2038565422]
}, Open  ]],

Cell["All three methods generate the same result.", "Text",
 CellID->1214943316],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"res1", "\[Equal]", "res2", "\[Equal]", "res3"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->1277796336],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->1086549152]
}, Open  ]],

Cell["\<\
The time for the overall process will scale with the square of the length of \
the list for the technique that has to copy at every step. With no copying \
the time will scale linearly. The longer the input list, the greater the \
impact, and the performance of the code will start to degrade.\
\>", "Text",
 CellID->1578147223],

Cell[CellGroupData[{

Cell["MORE ABOUT", "TutorialMoreAboutSection",
 CellID->98646871],

Cell[TextData[ButtonBox["Matrices and Linear Algebra",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatricesAndLinearAlgebra"]], "TutorialMoreAbout",
 CellID->4791866],

Cell[TextData[ButtonBox["Linear Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/LinearSystems"]], "TutorialMoreAbout",
 CellID->212289775]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->429810382],

Cell[TextData[{
 ButtonBox["Linear Algebra in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/LinearAlgebraOverview"],
  FontSlant->"Italic"]
}], "RelatedTutorials",
 CellID->315063104],

Cell[TextData[ButtonBox["Matrix and Tensor Operations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/LinearAlgebraMatrixAndTensorOperations"]], \
"RelatedTutorials",
 CellID->3571032],

Cell[TextData[ButtonBox["Working with Sparse Arrays",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/LinearAlgebraSparseArrays"]], "RelatedTutorials",\

 CellID->560755906],

Cell[TextData[ButtonBox["Matrix Computations",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/LinearAlgebraMatrixComputations"]], "RelatedTutorials",
 CellID->335456077],

Cell[TextData[ButtonBox["Matrix Types",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/LinearAlgebraMatrixTypes"]], "RelatedTutorials",\

 CellID->472696132]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Performance of Linear Algebra Computation - Wolfram \
Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 38, 42.1164992}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "This tutorial covers issues related to performance. One of the \
challenges of building Mathematica is to make sure that the system is general \
enough to support symbolic computation and fast enough so that numerical \
computation is efficient. These and other goals are described under \"Design \
Principles of Mathematica\". One technique that Mathematica provides is to \
use specialized representations for important types of Mathematica \
expression. In the case of linear algebra this involves what is known as \
Packed Array Technology, and it was first introduced in Mathematica 4.0. An \
example shows a vector of real numbers.", "synonyms" -> {}, "title" -> 
    "Performance of Linear Algebra Computation", "type" -> "Tutorial", "uri" -> 
    "tutorial/LinearAlgebraPerformance"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "t:59"->{
  Cell[3572, 94, 132, 3, 70, "Title",
   CellTags->{"t:59", "c:131"},
   CellID->195254286]},
 "c:131"->{
  Cell[3572, 94, 132, 3, 70, "Title",
   CellTags->{"t:59", "c:131"},
   CellID->195254286]},
 "t:109"->{
  Cell[3819, 104, 84, 2, 70, "Section",
   CellTags->{"t:109", "c:132"},
   CellID->1945433762]},
 "c:132"->{
  Cell[3819, 104, 84, 2, 70, "Section",
   CellTags->{"t:109", "c:132"},
   CellID->1945433762]},
 "S3.1.1"->{
  Cell[8000, 266, 892, 25, 70, "DefinitionBox",
   CellTags->"S3.1.1",
   CellID->277825037]},
 "S2.14.3"->{
  Cell[8895, 293, 81, 2, 70, "Caption",
   CellTags->"S2.14.3",
   CellID->775948281],
  Cell[12619, 447, 1404, 50, 70, "DefinitionBox",
   CellTags->"S2.14.3",
   CellID->767419931],
  Cell[14026, 499, 84, 2, 70, "Caption",
   CellTags->"S2.14.3",
   CellID->239996435],
  Cell[24205, 894, 701, 24, 70, "DefinitionBox",
   CellTags->"S2.14.3",
   CellID->1936617362],
  Cell[24909, 920, 151, 7, 70, "Caption",
   CellTags->"S2.14.3",
   CellID->1465849202],
  Cell[26207, 978, 836, 28, 70, "DefinitionBox",
   CellTags->"S2.14.3",
   CellID->1744557297],
  Cell[27046, 1008, 100, 2, 70, "Caption",
   CellTags->"S2.14.3",
   CellID->1180258188]},
 "t:177"->{
  Cell[12399, 438, 95, 2, 70, "Subsection",
   CellTags->{"t:177", "c:133"},
   CellID->995975105]},
 "c:133"->{
  Cell[12399, 438, 95, 2, 70, "Subsection",
   CellTags->{"t:177", "c:133"},
   CellID->995975105]},
 "t:182"->{
  Cell[17521, 646, 95, 2, 70, "Subsection",
   CellTags->{"t:182", "c:134"},
   CellID->41338579]},
 "c:134"->{
  Cell[17521, 646, 95, 2, 70, "Subsection",
   CellTags->{"t:182", "c:134"},
   CellID->41338579]},
 "t:183"->{
  Cell[21896, 810, 93, 2, 70, "Subsection",
   CellTags->{"t:183", "c:135"},
   CellID->169320987]},
 "c:135"->{
  Cell[21896, 810, 93, 2, 70, "Subsection",
   CellTags->{"t:183", "c:135"},
   CellID->169320987]},
 "t:108"->{
  Cell[23560, 868, 93, 2, 70, "Section",
   CellTags->{"t:108", "c:136"},
   CellID->1030559832],
  Cell[67136, 2487, 92, 2, 70, "Section",
   CellTags->{"t:108", "c:149"},
   CellID->1857660297]},
 "c:136"->{
  Cell[23560, 868, 93, 2, 70, "Section",
   CellTags->{"t:108", "c:136"},
   CellID->1030559832]},
 "t:161"->{
  Cell[23888, 882, 95, 2, 70, "Subsection",
   CellTags->{"t:161", "c:137"},
   CellID->1826376848]},
 "c:137"->{
  Cell[23888, 882, 95, 2, 70, "Subsection",
   CellTags->{"t:161", "c:137"},
   CellID->1826376848]},
 "t:162"->{
  Cell[28001, 1048, 91, 2, 70, "Subsection",
   CellTags->{"t:162", "c:138"},
   CellID->1352766973]},
 "c:138"->{
  Cell[28001, 1048, 91, 2, 70, "Subsection",
   CellTags->{"t:162", "c:138"},
   CellID->1352766973]},
 "t:163"->{
  Cell[28804, 1073, 90, 2, 70, "Subsubsection",
   CellTags->{"t:163", "c:139"},
   CellID->1048970201]},
 "c:139"->{
  Cell[28804, 1073, 90, 2, 70, "Subsubsection",
   CellTags->{"t:163", "c:139"},
   CellID->1048970201]},
 "S3.7.2"->{
  Cell[29604, 1104, 91, 2, 70, "Caption",
   CellTags->"S3.7.2",
   CellID->245796398],
  Cell[29934, 1120, 94, 2, 70, "Caption",
   CellTags->"S3.7.2",
   CellID->1084841352],
  Cell[30404, 1140, 101, 2, 70, "Caption",
   CellTags->"S3.7.2",
   CellID->662403043],
  Cell[31845, 1196, 88, 2, 70, "Caption",
   CellTags->"S3.7.2",
   CellID->2012713480],
  Cell[32151, 1209, 90, 2, 70, "Caption",
   CellTags->"S3.7.2",
   CellID->1163263041],
  Cell[32952, 1242, 88, 2, 70, "Caption",
   CellTags->"S3.7.2",
   CellID->1583230038],
  Cell[33455, 1264, 90, 2, 70, "Caption",
   CellTags->"S3.7.2",
   CellID->2087020600]},
 "t:164"->{
  Cell[31084, 1168, 89, 2, 70, "Subsubsection",
   CellTags->{"t:164", "c:140"},
   CellID->291032650]},
 "c:140"->{
  Cell[31084, 1168, 89, 2, 70, "Subsubsection",
   CellTags->{"t:164", "c:140"},
   CellID->291032650]},
 "t:165"->{
  Cell[33843, 1282, 96, 2, 70, "Subsection",
   CellTags->{"t:165", "c:141"},
   CellID->1145089169]},
 "c:141"->{
  Cell[33843, 1282, 96, 2, 70, "Subsection",
   CellTags->{"t:165", "c:141"},
   CellID->1145089169]},
 "t:166"->{
  Cell[37603, 1410, 87, 2, 70, "Subsubsection",
   CellTags->{"t:166", "c:142"},
   CellID->1855711455]},
 "c:142"->{
  Cell[37603, 1410, 87, 2, 70, "Subsubsection",
   CellTags->{"t:166", "c:142"},
   CellID->1855711455]},
 "t:167"->{
  Cell[43473, 1610, 88, 2, 70, "Subsubsection",
   CellTags->{"t:167", "c:143"},
   CellID->460055094]},
 "c:143"->{
  Cell[43473, 1610, 88, 2, 70, "Subsubsection",
   CellTags->{"t:167", "c:143"},
   CellID->460055094]},
 "t:168"->{
  Cell[48752, 1798, 96, 2, 70, "Subsubsection",
   CellTags->{"t:168", "c:144"},
   CellID->663127012]},
 "c:144"->{
  Cell[48752, 1798, 96, 2, 70, "Subsubsection",
   CellTags->{"t:168", "c:144"},
   CellID->663127012]},
 "t:184"->{
  Cell[53046, 1962, 88, 2, 70, "Subsection",
   CellTags->{"t:184", "c:145"},
   CellID->478605279]},
 "c:145"->{
  Cell[53046, 1962, 88, 2, 70, "Subsection",
   CellTags->{"t:184", "c:145"},
   CellID->478605279]},
 "t:178"->{
  Cell[54218, 1998, 108, 2, 70, "Subsubsection",
   CellTags->{"t:178", "c:146"},
   CellID->92425055]},
 "c:146"->{
  Cell[54218, 1998, 108, 2, 70, "Subsubsection",
   CellTags->{"t:178", "c:146"},
   CellID->92425055]},
 "t:185"->{
  Cell[56393, 2082, 106, 2, 70, "Subsubsection",
   CellTags->{"t:185", "c:147"},
   CellID->1705346414]},
 "c:147"->{
  Cell[56393, 2082, 106, 2, 70, "Subsubsection",
   CellTags->{"t:185", "c:147"},
   CellID->1705346414]},
 "t:186"->{
  Cell[62483, 2308, 92, 2, 70, "Subsubsection",
   CellTags->{"t:186", "c:148"},
   CellID->535763928]},
 "c:148"->{
  Cell[62483, 2308, 92, 2, 70, "Subsubsection",
   CellTags->{"t:186", "c:148"},
   CellID->535763928]},
 "c:149"->{
  Cell[67136, 2487, 92, 2, 70, "Section",
   CellTags->{"t:108", "c:149"},
   CellID->1857660297]},
 "t:187"->{
  Cell[68122, 2518, 94, 2, 70, "Subsection",
   CellTags->{"t:187", "c:150"},
   CellID->1264346631]},
 "c:150"->{
  Cell[68122, 2518, 94, 2, 70, "Subsection",
   CellTags->{"t:187", "c:150"},
   CellID->1264346631]},
 "t:188"->{
  Cell[74400, 2746, 95, 2, 70, "Subsection",
   CellTags->{"t:188", "c:151"},
   CellID->1750402393]},
 "c:151"->{
  Cell[74400, 2746, 95, 2, 70, "Subsection",
   CellTags->{"t:188", "c:151"},
   CellID->1750402393]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"t:59", 84572, 3116},
 {"c:131", 84678, 3120},
 {"t:109", 84784, 3124},
 {"c:132", 84894, 3128},
 {"S3.1.1", 85005, 3132},
 {"S2.14.3", 85114, 3136},
 {"t:177", 85774, 3158},
 {"c:133", 85887, 3162},
 {"t:182", 86000, 3166},
 {"c:134", 86112, 3170},
 {"t:183", 86224, 3174},
 {"c:135", 86337, 3178},
 {"t:108", 86450, 3182},
 {"c:136", 86660, 3189},
 {"t:161", 86771, 3193},
 {"c:137", 86885, 3197},
 {"t:162", 86999, 3201},
 {"c:138", 87114, 3205},
 {"t:163", 87229, 3209},
 {"c:139", 87347, 3213},
 {"S3.7.2", 87466, 3217},
 {"t:164", 88101, 3239},
 {"c:140", 88218, 3243},
 {"t:165", 88335, 3247},
 {"c:141", 88450, 3251},
 {"t:166", 88565, 3255},
 {"c:142", 88683, 3259},
 {"t:167", 88801, 3263},
 {"c:143", 88918, 3267},
 {"t:168", 89035, 3271},
 {"c:144", 89152, 3275},
 {"t:184", 89269, 3279},
 {"c:145", 89383, 3283},
 {"t:178", 89497, 3287},
 {"c:146", 89614, 3291},
 {"t:185", 89731, 3295},
 {"c:147", 89850, 3299},
 {"t:186", 89969, 3303},
 {"c:148", 90086, 3307},
 {"c:149", 90203, 3311},
 {"t:187", 90315, 3315},
 {"c:150", 90430, 3319},
 {"t:188", 90545, 3323},
 {"c:151", 90660, 3327}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 2947, 67, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3572, 94, 132, 3, 70, "Title",
 CellTags->{"t:59", "c:131"},
 CellID->195254286],
Cell[3707, 99, 87, 1, 70, "Text",
 CellID->283772186],
Cell[CellGroupData[{
Cell[3819, 104, 84, 2, 70, "Section",
 CellTags->{"t:109", "c:132"},
 CellID->1945433762],
Cell[3906, 108, 1006, 26, 70, "Text",
 CellID->1466712359],
Cell[4915, 136, 77, 1, 70, "Text",
 CellID->69833800],
Cell[CellGroupData[{
Cell[5017, 141, 278, 9, 47, "Input",
 CellID->1170605338],
Cell[5298, 152, 163, 5, 36, "Output",
 CellID->1266225599]
}, Open  ]],
Cell[CellGroupData[{
Cell[5498, 162, 205, 7, 28, "Input",
 CellID->2038184270],
Cell[5706, 171, 259, 9, 36, "Output",
 CellID->1251372593]
}, Open  ]],
Cell[5980, 183, 319, 9, 70, "Text",
 CellID->1511815480],
Cell[CellGroupData[{
Cell[6324, 196, 145, 5, 28, "Input",
 CellID->56261056],
Cell[6472, 203, 158, 5, 36, "Output",
 CellID->686763512]
}, Open  ]],
Cell[6645, 211, 116, 3, 70, "Text",
 CellID->1857227206],
Cell[CellGroupData[{
Cell[6786, 218, 103, 3, 70, "Input",
 CellID->742842097],
Cell[6892, 223, 164, 5, 36, "Output",
 CellID->1624684268]
}, Open  ]],
Cell[7071, 231, 111, 3, 70, "Text",
 CellID->813920671],
Cell[CellGroupData[{
Cell[7207, 238, 203, 7, 70, "Input",
 CellID->93886107],
Cell[7413, 247, 242, 8, 36, "Output",
 CellID->1690803451]
}, Open  ]],
Cell[7670, 258, 327, 6, 70, "Text",
 CellID->1829983853],
Cell[8000, 266, 892, 25, 70, "DefinitionBox",
 CellTags->"S3.1.1",
 CellID->277825037],
Cell[8895, 293, 81, 2, 70, "Caption",
 CellTags->"S2.14.3",
 CellID->775948281],
Cell[8979, 297, 416, 12, 70, "Text",
 CellID->1842054183],
Cell[CellGroupData[{
Cell[9420, 313, 299, 9, 70, "Input",
 CellID->1280398529],
Cell[9722, 324, 162, 5, 36, "Output",
 CellID->1871933963]
}, Open  ]],
Cell[9899, 332, 116, 3, 70, "Text",
 CellID->958687591],
Cell[CellGroupData[{
Cell[10040, 339, 231, 7, 70, "Input",
 CellID->92538496],
Cell[10274, 348, 162, 5, 36, "Output",
 CellID->516341844]
}, Open  ]],
Cell[10451, 356, 113, 3, 70, "Text",
 CellID->175954581],
Cell[CellGroupData[{
Cell[10589, 363, 204, 6, 70, "Input",
 CellID->183660948],
Cell[10796, 371, 162, 5, 36, "Output",
 CellID->855552297]
}, Open  ]],
Cell[10973, 379, 147, 4, 70, "Text",
 CellID->1864439014],
Cell[CellGroupData[{
Cell[11145, 387, 231, 7, 70, "Input",
 CellID->1692614921],
Cell[11379, 396, 162, 5, 36, "Output",
 CellID->63105438]
}, Open  ]],
Cell[11556, 404, 207, 5, 70, "Text",
 CellID->1903658495],
Cell[CellGroupData[{
Cell[11788, 413, 407, 13, 70, "Input",
 CellID->1576952459],
Cell[12198, 428, 164, 5, 36, "Output",
 CellID->1744290786]
}, Open  ]],
Cell[CellGroupData[{
Cell[12399, 438, 95, 2, 70, "Subsection",
 CellTags->{"t:177", "c:133"},
 CellID->995975105],
Cell[12497, 442, 119, 3, 70, "Text",
 CellID->1866045017],
Cell[12619, 447, 1404, 50, 70, "DefinitionBox",
 CellTags->"S2.14.3",
 CellID->767419931],
Cell[14026, 499, 84, 2, 70, "Caption",
 CellTags->"S2.14.3",
 CellID->239996435],
Cell[14113, 503, 149, 5, 70, "Text",
 CellID->645793979],
Cell[CellGroupData[{
Cell[14287, 512, 294, 8, 70, "Input",
 CellID->477053115],
Cell[14584, 522, 162, 5, 36, "Output",
 CellID->1275249554]
}, Open  ]],
Cell[14761, 530, 142, 4, 70, "Text",
 CellID->1232457018],
Cell[CellGroupData[{
Cell[14928, 538, 297, 8, 70, "Input",
 CellID->1425326211],
Cell[15228, 548, 162, 5, 36, "Output",
 CellID->277545992]
}, Open  ]],
Cell[15405, 556, 252, 8, 70, "Text",
 CellID->381006260],
Cell[CellGroupData[{
Cell[15682, 568, 330, 10, 70, "Input",
 CellID->1923460113],
Cell[16015, 580, 162, 5, 36, "Output",
 CellID->1522780900]
}, Open  ]],
Cell[16192, 588, 132, 5, 70, "Text",
 CellID->1815965001],
Cell[CellGroupData[{
Cell[16349, 597, 124, 3, 70, "Input",
 CellID->886180643],
Cell[16476, 602, 318, 9, 36, "Output",
 CellID->1238974956]
}, Open  ]],
Cell[16809, 614, 130, 4, 70, "Text",
 CellID->1450644180],
Cell[CellGroupData[{
Cell[16964, 622, 342, 11, 70, "Input",
 CellID->594888869],
Cell[17309, 635, 163, 5, 36, "Output",
 CellID->1760844498]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[17521, 646, 95, 2, 70, "Subsection",
 CellTags->{"t:182", "c:134"},
 CellID->41338579],
Cell[17619, 650, 611, 18, 70, "Text",
 CellID->1140279272],
Cell[CellGroupData[{
Cell[18255, 672, 322, 10, 70, "Input",
 CellID->529658676],
Cell[18580, 684, 161, 5, 36, "Output",
 CellID->653650717]
}, Open  ]],
Cell[18756, 692, 126, 4, 70, "Text",
 CellID->1528746214],
Cell[CellGroupData[{
Cell[18907, 700, 191, 7, 70, "Input",
 CellID->1143265990],
Cell[19101, 709, 162, 5, 36, "Output",
 CellID->1335304221]
}, Open  ]],
Cell[CellGroupData[{
Cell[19300, 719, 133, 4, 70, "Input",
 CellID->1253931197],
Cell[19436, 725, 162, 5, 36, "Output",
 CellID->1564599968]
}, Open  ]],
Cell[19613, 733, 490, 8, 70, "Text",
 CellID->294404661],
Cell[20106, 743, 120, 3, 70, "Text",
 CellID->256814739],
Cell[CellGroupData[{
Cell[20251, 750, 351, 9, 70, "Input",
 CellID->1990671550],
Cell[20605, 761, 162, 5, 36, "Output",
 CellID->1569004237]
}, Open  ]],
Cell[20782, 769, 220, 5, 70, "Text",
 CellID->1302509746],
Cell[CellGroupData[{
Cell[21027, 778, 167, 6, 70, "Input",
 CellID->1980487151],
Cell[21197, 786, 349, 7, 70, "Message",
 CellID->51018357]
}, Open  ]],
Cell[21561, 796, 92, 1, 70, "Text",
 CellID->619550946],
Cell[21656, 799, 203, 6, 70, "Input",
 CellID->1792028766]
}, Open  ]],
Cell[CellGroupData[{
Cell[21896, 810, 93, 2, 70, "Subsection",
 CellTags->{"t:183", "c:135"},
 CellID->169320987],
Cell[21992, 814, 611, 15, 70, "Text",
 CellID->361996234],
Cell[CellGroupData[{
Cell[22628, 833, 374, 12, 70, "Input",
 CellID->249532887],
Cell[23005, 847, 162, 5, 36, "Output",
 CellID->2011152865]
}, Open  ]],
Cell[23182, 855, 329, 7, 70, "Text",
 CellID->53598275]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[23560, 868, 93, 2, 70, "Section",
 CellTags->{"t:108", "c:136"},
 CellID->1030559832],
Cell[23656, 872, 207, 6, 70, "Text",
 CellID->1163483369],
Cell[CellGroupData[{
Cell[23888, 882, 95, 2, 70, "Subsection",
 CellTags->{"t:161", "c:137"},
 CellID->1826376848],
Cell[23986, 886, 216, 6, 70, "Text",
 CellID->1931917765],
Cell[24205, 894, 701, 24, 70, "DefinitionBox",
 CellTags->"S2.14.3",
 CellID->1936617362],
Cell[24909, 920, 151, 7, 70, "Caption",
 CellTags->"S2.14.3",
 CellID->1465849202],
Cell[25063, 929, 277, 10, 70, "Text",
 CellID->595139630],
Cell[CellGroupData[{
Cell[25365, 943, 361, 12, 70, "Input",
 CellID->1680417331],
Cell[25729, 957, 242, 8, 36, "Output",
 CellID->1216640420]
}, Open  ]],
Cell[25986, 968, 218, 8, 70, "Text",
 CellID->1883735853],
Cell[26207, 978, 836, 28, 70, "DefinitionBox",
 CellTags->"S2.14.3",
 CellID->1744557297],
Cell[27046, 1008, 100, 2, 70, "Caption",
 CellTags->"S2.14.3",
 CellID->1180258188],
Cell[27149, 1012, 146, 4, 70, "Text",
 CellID->292918377],
Cell[CellGroupData[{
Cell[27320, 1020, 464, 15, 70, "Input",
 CellID->1046643315],
Cell[27787, 1037, 165, 5, 36, "Output",
 CellID->895642628]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28001, 1048, 91, 2, 70, "Subsection",
 CellTags->{"t:162", "c:138"},
 CellID->1352766973],
Cell[28095, 1052, 213, 5, 70, "Text",
 CellID->545144422],
Cell[28311, 1059, 468, 10, 70, "Text",
 CellID->1077062452],
Cell[CellGroupData[{
Cell[28804, 1073, 90, 2, 70, "Subsubsection",
 CellTags->{"t:163", "c:139"},
 CellID->1048970201],
Cell[28897, 1077, 154, 4, 70, "Text",
 CellID->1348909984],
Cell[29054, 1083, 119, 2, 70, "Program",
 CellID->145239709],
Cell[29176, 1087, 292, 8, 70, "Text",
 CellID->1502366171],
Cell[29471, 1097, 130, 5, 70, "Program",
 CellID->155514571],
Cell[29604, 1104, 91, 2, 70, "Caption",
 CellTags->"S3.7.2",
 CellID->245796398],
Cell[29698, 1108, 134, 4, 70, "Text",
 CellID->1489008005],
Cell[29835, 1114, 96, 4, 70, "Program",
 CellID->1130601544],
Cell[29934, 1120, 94, 2, 70, "Caption",
 CellTags->"S3.7.2",
 CellID->1084841352],
Cell[30031, 1124, 279, 8, 70, "Text",
 CellID->792440034],
Cell[30313, 1134, 88, 4, 70, "Program",
 CellID->1993556391],
Cell[30404, 1140, 101, 2, 70, "Caption",
 CellTags->"S3.7.2",
 CellID->662403043],
Cell[30508, 1144, 539, 19, 70, "Text",
 CellID->452717653]
}, Open  ]],
Cell[CellGroupData[{
Cell[31084, 1168, 89, 2, 70, "Subsubsection",
 CellTags->{"t:164", "c:140"},
 CellID->291032650],
Cell[31176, 1172, 268, 9, 70, "Text",
 CellID->1565484331],
Cell[31447, 1183, 161, 2, 70, "Program",
 CellID->125508073],
Cell[31611, 1187, 154, 4, 70, "Text",
 CellID->585241557],
Cell[31768, 1193, 74, 1, 70, "Program",
 CellID->679745239],
Cell[31845, 1196, 88, 2, 70, "Caption",
 CellTags->"S3.7.2",
 CellID->2012713480],
Cell[31936, 1200, 158, 4, 70, "Text",
 CellID->513050816],
Cell[32097, 1206, 51, 1, 70, "Program",
 CellID->698221807],
Cell[32151, 1209, 90, 2, 70, "Caption",
 CellTags->"S3.7.2",
 CellID->1163263041],
Cell[32244, 1213, 134, 4, 70, "Text",
 CellID->270807799],
Cell[32381, 1219, 305, 9, 70, "Program",
 CellID->948829800],
Cell[32689, 1230, 157, 4, 70, "Text",
 CellID->1916369625],
Cell[32849, 1236, 100, 4, 70, "Program",
 CellID->464549669],
Cell[32952, 1242, 88, 2, 70, "Caption",
 CellTags->"S3.7.2",
 CellID->1583230038],
Cell[33043, 1246, 132, 4, 70, "Text",
 CellID->2024862336],
Cell[33178, 1252, 274, 10, 70, "Program",
 CellID->687927143],
Cell[33455, 1264, 90, 2, 70, "Caption",
 CellTags->"S3.7.2",
 CellID->2087020600],
Cell[33548, 1268, 246, 8, 70, "Text",
 CellID->1704817749]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[33843, 1282, 96, 2, 70, "Subsection",
 CellTags->{"t:165", "c:141"},
 CellID->1145089169],
Cell[33942, 1286, 514, 10, 70, "Text",
 CellID->1912702616],
Cell[34459, 1298, 647, 16, 70, "Text",
 CellID->1110155169],
Cell[35109, 1316, 1589, 54, 70, "Text",
 CellID->1040000247],
Cell[36701, 1372, 183, 4, 70, "Text",
 CellID->914748396],
Cell[36887, 1378, 160, 8, 70, "Program",
 CellID->340862699],
Cell[37050, 1388, 89, 1, 70, "Text",
 CellID->1540495544],
Cell[37142, 1391, 325, 10, 70, "Program",
 CellID->1045759776],
Cell[37470, 1403, 108, 3, 70, "Text",
 CellID->1078388184],
Cell[CellGroupData[{
Cell[37603, 1410, 87, 2, 70, "Subsubsection",
 CellTags->{"t:166", "c:142"},
 CellID->1855711455],
Cell[37693, 1414, 154, 4, 70, "Text",
 CellID->1037418804],
Cell[37850, 1420, 1961, 55, 70, "Input",
 CellID->478108835],
Cell[CellGroupData[{
Cell[39836, 1479, 411, 14, 70, "Input",
 CellID->1838866418],
Cell[40250, 1495, 505, 14, 92, "Output",
 CellID->1390723349]
}, Open  ]],
Cell[CellGroupData[{
Cell[40792, 1514, 408, 14, 70, "Input",
 CellID->675312695],
Cell[41203, 1530, 241, 8, 36, "Output",
 CellID->529390470]
}, Open  ]],
Cell[41459, 1541, 449, 10, 70, "Text",
 CellID->440879769],
Cell[41911, 1553, 186, 4, 70, "Text",
 CellID->1047695433],
Cell[CellGroupData[{
Cell[42122, 1561, 554, 18, 70, "Input",
 CellID->1398720678],
Cell[42679, 1581, 745, 23, 54, "Output",
 CellID->1942435084]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[43473, 1610, 88, 2, 70, "Subsubsection",
 CellTags->{"t:167", "c:143"},
 CellID->460055094],
Cell[43564, 1614, 193, 6, 70, "Text",
 CellID->1204835658],
Cell[43760, 1622, 1279, 36, 70, "Input",
 CellID->1049596443],
Cell[CellGroupData[{
Cell[45064, 1662, 377, 12, 70, "Input",
 CellID->1136741043],
Cell[45444, 1676, 865, 23, 54, "Output",
 CellID->1192185989]
}, Open  ]],
Cell[CellGroupData[{
Cell[46346, 1704, 385, 13, 70, "Input",
 CellID->1047490613],
Cell[46734, 1719, 258, 9, 36, "Output",
 CellID->103794585]
}, Open  ]],
Cell[47007, 1731, 452, 14, 70, "Text",
 CellID->718868731],
Cell[47462, 1747, 176, 4, 70, "Text",
 CellID->283763895],
Cell[CellGroupData[{
Cell[47663, 1755, 556, 18, 70, "Input",
 CellID->1806086925],
Cell[48222, 1775, 481, 17, 36, "Output",
 CellID->221598130]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[48752, 1798, 96, 2, 70, "Subsubsection",
 CellTags->{"t:168", "c:144"},
 CellID->663127012],
Cell[48851, 1802, 277, 9, 70, "Text",
 CellID->1883046401],
Cell[CellGroupData[{
Cell[49153, 1815, 420, 14, 70, "Input",
 CellID->894355768],
Cell[49576, 1831, 745, 23, 54, "Output",
 CellID->1199347423]
}, Open  ]],
Cell[CellGroupData[{
Cell[50358, 1859, 436, 15, 70, "Input",
 CellID->1150627619],
Cell[50797, 1876, 258, 9, 36, "Output",
 CellID->508151298]
}, Open  ]],
Cell[51070, 1888, 627, 18, 70, "Text",
 CellID->2103885014],
Cell[51700, 1908, 177, 4, 70, "Text",
 CellID->1677182443],
Cell[CellGroupData[{
Cell[51902, 1916, 598, 20, 70, "Input",
 CellID->534968979],
Cell[52503, 1938, 482, 17, 36, "Output",
 CellID->1857083711]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[53046, 1962, 88, 2, 70, "Subsection",
 CellTags->{"t:184", "c:145"},
 CellID->478605279],
Cell[53137, 1966, 614, 16, 70, "Text",
 CellID->1592666138],
Cell[53754, 1984, 439, 10, 70, "Text",
 CellID->1703605158],
Cell[CellGroupData[{
Cell[54218, 1998, 108, 2, 70, "Subsubsection",
 CellTags->{"t:178", "c:146"},
 CellID->92425055],
Cell[54329, 2002, 493, 10, 70, "Text",
 CellID->757222604],
Cell[54825, 2014, 200, 5, 70, "Text",
 CellID->652213306],
Cell[CellGroupData[{
Cell[55050, 2023, 354, 12, 70, "Input",
 CellID->992499654],
Cell[55407, 2037, 260, 9, 36, "Output",
 CellID->1436198529]
}, Open  ]],
Cell[CellGroupData[{
Cell[55704, 2051, 279, 10, 70, "Input",
 CellID->472484623],
Cell[55986, 2063, 242, 8, 36, "Output",
 CellID->1802015802]
}, Open  ]],
Cell[56243, 2074, 113, 3, 70, "Text",
 CellID->517710573]
}, Open  ]],
Cell[CellGroupData[{
Cell[56393, 2082, 106, 2, 70, "Subsubsection",
 CellTags->{"t:185", "c:147"},
 CellID->1705346414],
Cell[56502, 2086, 670, 15, 70, "Text",
 CellID->1440427236],
Cell[57175, 2103, 301, 6, 70, "Text",
 CellID->677402821],
Cell[CellGroupData[{
Cell[57501, 2113, 355, 12, 70, "Input",
 CellID->1298399800],
Cell[57859, 2127, 241, 8, 36, "Output",
 CellID->1624101433]
}, Open  ]],
Cell[CellGroupData[{
Cell[58137, 2140, 280, 10, 70, "Input",
 CellID->1755631061],
Cell[58420, 2152, 258, 9, 36, "Output",
 CellID->465932447]
}, Open  ]],
Cell[58693, 2164, 620, 14, 70, "Text",
 CellID->1201259767],
Cell[CellGroupData[{
Cell[59338, 2182, 779, 26, 70, "Input",
 CellID->1079138522],
Cell[60120, 2210, 259, 9, 36, "Output",
 CellID->1581740339]
}, Open  ]],
Cell[CellGroupData[{
Cell[60416, 2224, 120, 3, 70, "Input",
 CellID->609811360],
Cell[60539, 2229, 163, 5, 36, "Output",
 CellID->1307564036]
}, Open  ]],
Cell[60717, 2237, 180, 4, 70, "Text",
 CellID->1028593301],
Cell[CellGroupData[{
Cell[60922, 2245, 775, 26, 70, "Input",
 CellID->324251557],
Cell[61700, 2273, 240, 8, 36, "Output",
 CellID->1529906060]
}, Open  ]],
Cell[CellGroupData[{
Cell[61977, 2286, 122, 3, 70, "Input",
 CellID->1309444889],
Cell[62102, 2291, 162, 5, 36, "Output",
 CellID->427834967]
}, Open  ]],
Cell[62279, 2299, 167, 4, 70, "Text",
 CellID->847040213]
}, Open  ]],
Cell[CellGroupData[{
Cell[62483, 2308, 92, 2, 70, "Subsubsection",
 CellTags->{"t:186", "c:148"},
 CellID->535763928],
Cell[62578, 2312, 809, 20, 70, "Text",
 CellID->185406025],
Cell[CellGroupData[{
Cell[63412, 2336, 457, 16, 70, "Input",
 CellID->120290378],
Cell[63872, 2354, 258, 9, 36, "Output",
 CellID->938271219]
}, Open  ]],
Cell[CellGroupData[{
Cell[64167, 2368, 541, 19, 70, "Input",
 CellID->551957812],
Cell[64711, 2389, 239, 8, 36, "Output",
 CellID->1773074891]
}, Open  ]],
Cell[64965, 2400, 301, 6, 70, "Text",
 CellID->1586607265],
Cell[CellGroupData[{
Cell[65291, 2410, 256, 9, 70, "Input",
 CellID->803858056],
Cell[65550, 2421, 434, 17, 51, "Output",
 CellID->622723915]
}, Open  ]],
Cell[CellGroupData[{
Cell[66021, 2443, 291, 10, 70, "Input",
 CellID->1433027574],
Cell[66315, 2455, 258, 8, 36, "Output",
 CellID->865335113]
}, Open  ]],
Cell[66588, 2466, 318, 8, 70, "Text",
 CellID->927986707],
Cell[66909, 2476, 166, 4, 70, "Text",
 CellID->1646340035]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[67136, 2487, 92, 2, 70, "Section",
 CellTags->{"t:108", "c:149"},
 CellID->1857660297],
Cell[67231, 2491, 866, 23, 70, "Text",
 CellID->1579351593],
Cell[CellGroupData[{
Cell[68122, 2518, 94, 2, 70, "Subsection",
 CellTags->{"t:187", "c:150"},
 CellID->1264346631],
Cell[68219, 2522, 730, 16, 70, "Text",
 CellID->523140152],
Cell[68952, 2540, 231, 8, 70, "Input",
 CellID->618483846],
Cell[69186, 2550, 75, 1, 70, "Text",
 CellID->1037148242],
Cell[CellGroupData[{
Cell[69286, 2555, 185, 7, 70, "Input",
 CellID->225740595],
Cell[69474, 2564, 214, 7, 36, "Output",
 CellID->1566059563]
}, Open  ]],
Cell[69703, 2574, 151, 4, 70, "Text",
 CellID->1203497709],
Cell[CellGroupData[{
Cell[69879, 2582, 72, 2, 70, "Input",
 CellID->37524070],
Cell[69954, 2586, 214, 7, 36, "Output",
 CellID->1454335919]
}, Open  ]],
Cell[70183, 2596, 460, 8, 70, "Text",
 CellID->364920479],
Cell[CellGroupData[{
Cell[70668, 2608, 473, 17, 70, "Input",
 CellID->5194721],
Cell[71144, 2627, 259, 9, 36, "Output",
 CellID->1408329967]
}, Open  ]],
Cell[71418, 2639, 172, 4, 70, "Text",
 CellID->236998135],
Cell[CellGroupData[{
Cell[71615, 2647, 561, 19, 70, "Input",
 CellID->1792973407],
Cell[72179, 2668, 257, 9, 36, "Output",
 CellID->1013516745]
}, Open  ]],
Cell[72451, 2680, 265, 8, 70, "Text",
 CellID->335645211],
Cell[CellGroupData[{
Cell[72741, 2692, 503, 17, 70, "Input",
 CellID->1897004255],
Cell[73247, 2711, 258, 9, 36, "Output",
 CellID->1408290423]
}, Open  ]],
Cell[73520, 2723, 503, 10, 70, "Text",
 CellID->1719865774],
Cell[74026, 2735, 337, 6, 70, "Text",
 CellID->654174998]
}, Open  ]],
Cell[CellGroupData[{
Cell[74400, 2746, 95, 2, 70, "Subsection",
 CellTags->{"t:188", "c:151"},
 CellID->1750402393],
Cell[74498, 2750, 550, 13, 70, "Text",
 CellID->634538898],
Cell[CellGroupData[{
Cell[75073, 2767, 372, 14, 70, "Input",
 CellID->291467684],
Cell[75448, 2783, 257, 9, 36, "Output",
 CellID->287166181]
}, Open  ]],
Cell[75720, 2795, 109, 3, 70, "Text",
 CellID->1006249258],
Cell[CellGroupData[{
Cell[75854, 2802, 264, 10, 70, "Input",
 CellID->597559081],
Cell[76121, 2814, 239, 8, 36, "Output",
 CellID->1541940995]
}, Open  ]],
Cell[76375, 2825, 189, 4, 70, "Text",
 CellID->1717134616],
Cell[76567, 2831, 290, 10, 70, "Input",
 CellID->1996242726],
Cell[76860, 2843, 205, 8, 70, "Text",
 CellID->714478801],
Cell[CellGroupData[{
Cell[77090, 2855, 453, 16, 70, "Input",
 CellID->1664138030],
Cell[77546, 2873, 258, 9, 36, "Output",
 CellID->340851682]
}, Open  ]],
Cell[77819, 2885, 277, 9, 70, "Text",
 CellID->1355332825],
Cell[CellGroupData[{
Cell[78121, 2898, 403, 14, 70, "Input",
 CellID->719746046],
Cell[78527, 2914, 260, 9, 36, "Output",
 CellID->2143895218]
}, Open  ]],
Cell[78802, 2926, 711, 24, 70, "Text",
 CellID->1966299560],
Cell[CellGroupData[{
Cell[79538, 2954, 511, 18, 70, "Input",
 CellID->2118344884],
Cell[80052, 2974, 260, 9, 36, "Output",
 CellID->2038565422]
}, Open  ]],
Cell[80327, 2986, 80, 1, 70, "Text",
 CellID->1214943316],
Cell[CellGroupData[{
Cell[80432, 2991, 127, 3, 70, "Input",
 CellID->1277796336],
Cell[80562, 2996, 163, 5, 36, "Output",
 CellID->1086549152]
}, Open  ]],
Cell[80740, 3004, 338, 6, 70, "Text",
 CellID->1578147223],
Cell[CellGroupData[{
Cell[81103, 3014, 65, 1, 70, "TutorialMoreAboutSection",
 CellID->98646871],
Cell[81171, 3017, 169, 3, 70, "TutorialMoreAbout",
 CellID->4791866],
Cell[81343, 3022, 147, 3, 70, "TutorialMoreAbout",
 CellID->212289775]
}, Open  ]],
Cell[CellGroupData[{
Cell[81527, 3030, 72, 1, 70, "RelatedTutorialsSection",
 CellID->429810382],
Cell[81602, 3033, 304, 9, 70, "RelatedTutorials",
 CellID->315063104],
Cell[81909, 3044, 191, 5, 70, "RelatedTutorials",
 CellID->3571032],
Cell[82103, 3051, 175, 4, 70, "RelatedTutorials",
 CellID->560755906],
Cell[82281, 3057, 175, 4, 70, "RelatedTutorials",
 CellID->335456077],
Cell[82459, 3063, 160, 4, 70, "RelatedTutorials",
 CellID->472696132]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[82670, 3073, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

