(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    143333,       4149]
NotebookOptionsPosition[    130010,       3713]
NotebookOutlinePosition[    133314,       3800]
CellTagsIndexPosition[    133165,       3792]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Advanced Numerical Differential Equation Solving in \
Mathematica" :> 
         Documentation`HelpLookup["paclet:tutorial/NDSolveOverview"]}, #, 
        2]& , Evaluator -> Automatic], {
      1->"\<\"Advanced Numerical Differential Equation Solving in Mathematica\
\"\>"}, "\<\"related tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Numerical Solution of Boundary Value Problems", "Title",
 CellID->1974382719],

Cell[CellGroupData[{

Cell["Shooting Method", "Section",
 CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
 CellID->659822336],

Cell["\<\
The shooting method works by considering the boundary conditions as a \
multivariate function of initial conditions at some point, reducing the \
boundary value problem to finding the initial conditions that give a root. \
The advantage of the shooting method is that it takes advantage of the speed \
and adaptivity of methods for initial value problems. The disadvantage of the \
method is that it is not as robust as finite difference or collocation \
methods: some initial value problems with growing modes are inherently \
unstable even though the BVP itself may be quite well posed and stable.\
\>", "Text",
 CellID->96799541],

Cell[TextData[{
 "Consider the BVP system\n         ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"X", "'"}], 
       RowBox[{"(", "t", ")"}]}], " ", "=", " ", 
      RowBox[{"F", "(", 
       RowBox[{"t", ",", " ", 
        RowBox[{"X", "(", "t", ")"}]}], ")"}]}], ";", " ", 
     RowBox[{
      RowBox[{"G", "(", 
       RowBox[{
        RowBox[{"X", "(", 
         SubscriptBox["t", "1"], ")"}], ",", " ", 
        RowBox[{"X", "(", 
         SubscriptBox["t", "2"], ")"}], ",", " ", "...", ",", " ", 
        RowBox[{"X", "(", 
         SubscriptBox["t", "n"], ")"}]}], ")"}], " ", "=", " ", "0"}]}], ",", 
    " ", 
    RowBox[{
     RowBox[{
      RowBox[{
       SubscriptBox["t", "1"], "<", " ", 
       SubscriptBox["t", "2"], "<"}], " ", "..."}], " ", "<", " ", 
     SubscriptBox["t", "n"]}]}], TraditionalForm]], "InlineMath"]
}], "Text",
 CellID->382453414],

Cell[TextData[{
 "The shooting method looks for initial conditions ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"X", "(", 
     SubscriptBox["t", "0"], ")"}], " ", "=", " ", "c"}], TraditionalForm]], 
  "InlineMath"],
 " so that ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", " ", "=", " ", "0"}], TraditionalForm]], "InlineMath"],
 ".",
 " ",
 "Since we are varying the initial conditions, it makes sense to think of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"X", " ", "=", " ", 
    SubscriptBox["X", "c"]}], TraditionalForm]], "InlineMath"],
 " as a function of them, so shooting can be thought of as finding ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 " such that with "
}], "Text",
 CellID->177552007],

Cell[TextData[{
 "      ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       SubscriptBox["X", "c"], "'"}], 
      RowBox[{"(", "t", ")"}]}], " ", "=", " ", 
     RowBox[{"F", "(", 
      RowBox[{"t", ",", " ", 
       RowBox[{
        SubscriptBox["X", "c"], "(", "t", ")"}]}], ")"}]}], ";", " ", 
    RowBox[{
     RowBox[{
      SubscriptBox["X", "c"], "(", 
      SubscriptBox["t", "0"], ")"}], " ", "=", " ", "c"}]}], 
   TraditionalForm]], "InlineMath"],
 "\n      ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"G", "(", 
     RowBox[{
      RowBox[{
       SubscriptBox["X", "c"], "(", 
       SubscriptBox["t", "1"], ")"}], ",", " ", 
      RowBox[{
       SubscriptBox["X", "c"], "(", 
       SubscriptBox["t", "2"], ")"}], ",", " ", "...", ",", " ", 
      RowBox[{
       SubscriptBox["X", "c"], "(", 
       SubscriptBox["t", "n"], ")"}]}], ")"}], " ", "=", " ", "0"}], 
   TraditionalForm]], "InlineMath"]
}], "Text",
 CellID->254723824],

Cell[TextData[{
 "After setting up the function for ",
 Cell[BoxData[
  FormBox["G", TraditionalForm]], "InlineMath"],
 ", the problem is effectively passed to FindRoot to find the initial \
conditions ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 " giving the root. The default method is to use Newton's method, which \
involves computing the Jacobian.",
 " ",
 "While the Jacobian can be computed using finite differences, the \
sensitivity of solutions of an IVP to its initial conditions may be too much \
to get reasonably accurate derivative values, so, it is advantageous to \
compute the Jacobian as a solution to ODEs."
}], "Text",
 CellID->584806262],

Cell[CellGroupData[{

Cell["Linearization and Newton's Method", "Subsection",
 CellTags->"s:5",
 CellID->468358538],

Cell["Linear problems can be described by ", "Text",
 CellID->622958117],

Cell[TextData[{
 "     ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       SubscriptBox["X", "c"], "'"}], 
      RowBox[{"(", "t", ")"}]}], " ", "=", 
     RowBox[{
      RowBox[{
       RowBox[{"J", "(", "t", ")"}], " ", 
       RowBox[{
        SubscriptBox["X", "c"], "(", "t", ")"}]}], " ", "+", " ", 
      RowBox[{
       SubscriptBox["F", "0"], "(", "t", ")"}]}]}], ";", " ", 
    RowBox[{
     RowBox[{
      SubscriptBox["X", "c"], "(", 
      SubscriptBox["t", "0"], ")"}], " ", "=", " ", "c"}]}], 
   TraditionalForm]], "InlineMath"],
 "\n    ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"G", "(", 
     RowBox[{
      RowBox[{
       SubscriptBox["X", "c"], "(", 
       SubscriptBox["t", "1"], ")"}], ",", " ", 
      RowBox[{
       SubscriptBox["X", "c"], "(", 
       SubscriptBox["t", "2"], ")"}], ",", " ", "...", ",", " ", 
      RowBox[{
       SubscriptBox["X", "c"], "(", 
       SubscriptBox["t", "n"], ")"}]}], ")"}], " ", "=", " ", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["B", "0"], "+", 
        RowBox[{
         SubscriptBox["B", "1"], 
         RowBox[{
          SubscriptBox["X", "c"], "(", 
          SubscriptBox["t", "1"], ")"}]}], "+", 
        RowBox[{
         SubscriptBox["B", "2"], " ", 
         RowBox[{
          SubscriptBox["X", "c"], "(", 
          SubscriptBox["t", "2"], ")"}]}], " ", "+"}], " ", "..."}], ".", 
      "  ", 
      SubscriptBox["B", "n"]}], " ", 
     RowBox[{
      SubscriptBox["X", "c"], "(", 
      SubscriptBox["t", "n"], ")"}], " "}]}], TraditionalForm]], 
  "InlineMath"]
}], "Text",
 CellID->410269708],

Cell[TextData[{
 "Where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"J", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
 " is a matrix and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["F", "0"], "(", "t", ")"}], TraditionalForm]], 
  "InlineMath"],
 " is a vector both possibly depending on ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["B", "0"], TraditionalForm]], "InlineMath"],
 " is a constant vector, and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["B", "1"], ",", " ", 
    SubscriptBox["B", "2"], ",", " ", "...", " ", ",", " ", 
    SubscriptBox["B", "n"]}], TraditionalForm]], "InlineMath"],
 " are constant matrices."
}], "Text",
 CellID->115473309],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox[
   RowBox[{"Y", " ", "=", " "}], TraditionalForm]], "InlineMath"],
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"\[PartialD]", 
     RowBox[{
      SubscriptBox["X", "c"], "(", "t", ")"}]}], 
    RowBox[{"\[PartialD]", "c"}],
    MultilineFunction->None], TraditionalForm]], "InlineMath"],
 ".",
 " ",
 "Then, differentiating both the IVP and boundary conditions with respect to \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"c", " "}], TraditionalForm]], "InlineMath"],
 " gives"
}], "Text",
 CellID->206138752],

Cell[TextData[{
 "     ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Y", "'"}], 
       RowBox[{"(", "t", ")"}]}], " ", "=", 
      RowBox[{
       RowBox[{"J", "(", "t", ")"}], " ", 
       RowBox[{"Y", "(", "t", ")"}]}]}], ")"}], ";", " ", 
    RowBox[{
     RowBox[{"Y", "(", 
      SubscriptBox["t", "0"], ")"}], " ", "=", "I"}]}], TraditionalForm]], 
  "InlineMath"],
 "\n    ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox[
     RowBox[{"\[PartialD]", "G"}], 
     RowBox[{"\[PartialD]", "c"}],
     MultilineFunction->None], "=", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{
          SubscriptBox["B", "1"], 
          RowBox[{"Y", "(", 
           SubscriptBox["t", "1"], ")"}]}], "+", 
         RowBox[{
          SubscriptBox["B", "2"], " ", 
          RowBox[{"Y", "(", 
           SubscriptBox["t", "2"], ")"}]}], " ", "+"}], " ", "..."}], ".", 
       "  ", 
       SubscriptBox["B", "n"]}], " ", 
      RowBox[{"Y", "(", 
       SubscriptBox["t", "n"], ")"}]}], " ", "=", " ", "0"}]}], 
   TraditionalForm]], "InlineMath"]
}], "Text",
 CellID->26324775],

Cell[TextData[{
 "Since ",
 Cell[BoxData[
  FormBox["G", TraditionalForm]], "InlineMath"],
 " is linear, when thought of as a function of ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 ", we have ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      FormBox[
       RowBox[{
        RowBox[{"G", "(", "c", ")"}], " ", "=", " ", 
        RowBox[{
         RowBox[{"G", "(", 
          SubscriptBox["c", "0"], ")"}], " ", "+", " "}]}],
       TraditionalForm], 
      FormBox[
       RowBox[{"(", 
        FractionBox[
         RowBox[{"\[PartialD]", "G"}], 
         RowBox[{"\[PartialD]", "c"}],
         MultilineFunction->None]}],
       TraditionalForm]}], " ", ")"}], 
    RowBox[{"(", 
     RowBox[{"c", " ", "-", " ", 
      SubscriptBox["c", "0"]}], ")"}]}], TraditionalForm]], "InlineMath"],
 ", so the value of ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 " for which ",
 Cell[BoxData[
  FormBox[
   RowBox[{"G", "(", "c", ")"}], TraditionalForm]], "InlineMath"],
 " = 0 satisfies"
}], "Text",
 CellID->306888864],

Cell[BoxData[
 FormBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"         ", 
     RowBox[{
      FormBox[
       RowBox[{"c", "=", "  ", 
        RowBox[{
         SubscriptBox["c", "0"], "+", " "}]}],
       TraditionalForm], 
      FormBox[
       RowBox[{"(", 
        FractionBox[
         RowBox[{"\[PartialD]", "G"}], 
         RowBox[{"\[PartialD]", "c"}],
         MultilineFunction->None]}],
       TraditionalForm]}], " ", ")"}], 
    RowBox[{"-", "1"}]], 
   RowBox[{"G", "(", 
    SubscriptBox["c", "0"], ")"}]}], TraditionalForm]], "Text",
 CellID->544456068],

Cell[TextData[{
 "for any particular initial condition ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["c", "0"], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->180742921],

Cell[TextData[{
 "For nonlinear problems, let ",
 Cell[BoxData[
  FormBox[
   RowBox[{"J", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
 " be the Jacobian for the nonlinear ODE system, and let ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["B", "i"], TraditionalForm]], "InlineMath"],
 " be the Jacobian of the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["i", "th"], TraditionalForm]]],
 " boundary condition.",
 " ",
 "Then computation of ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"\[PartialD]", "G"}], 
    RowBox[{"\[PartialD]", "c"}],
    MultilineFunction->None], TraditionalForm]], "InlineMath"],
 "for the linearized system gives the Jacobian for the nonlinear system for a \
particular initial condition, leading to a Newton iteration,"
}], "Text",
 CellID->45412943],

Cell[BoxData[
 FormBox[
  RowBox[{
   SuperscriptBox[
    RowBox[{"         ", 
     RowBox[{
      FormBox[
       RowBox[{
        SubscriptBox["c", 
         RowBox[{"n", "+", "1"}]], "=", "  ", 
        RowBox[{
         SubscriptBox["c", "n"], "+", " "}]}],
       TraditionalForm], 
      FormBox[
       RowBox[{"(", 
        RowBox[{
         FractionBox[
          RowBox[{"\[PartialD]", "G"}], 
          RowBox[{"\[PartialD]", "c"}],
          MultilineFunction->None], 
         RowBox[{"(", 
          SubscriptBox["c", "n"], ")"}]}]}],
       TraditionalForm]}], " ", ")"}], 
    RowBox[{"-", "1"}]], 
   RowBox[{"G", "(", 
    SubscriptBox["c", "n"], ")"}]}], TraditionalForm]], "Text",
 CellID->241406211]
}, Open  ]],

Cell[CellGroupData[{

Cell["StartingInitialConditions", "Subsection",
 CellTags->"s:5",
 CellID->224929489],

Cell["\<\
For boundary value problems, there is no guarantee of uniqueness as there is \
in the initial value problem case. Shooting will find only one solution. Just \
as you can affect the particular solution FindRoot gets for a system of \
nonlinear algebraic equations by changing the starting values, you can change \
the solution that shooting finds by giving different initial conditions to \
start the iterations from.\
\>", "Text",
 CellID->19996186],

Cell[TextData[{
 StyleBox["StartingInitialConditions", "MR"],
 " is an option of the ",
 StyleBox["Shooting", "MR"],
 " method that allows you to specify the values and position of the initial \
conditions to start the shooting process from."
}], "Text",
 CellID->159580041],

Cell["\<\
The shooting method by default starts with zero initial conditions so that if \
there is a zero solution, it will be returned.\
\>", "Text",
 CellID->113392259],

Cell[TextData[{
 "This computes a very simple solution to the boundary value problem ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "''"}], "+", 
      RowBox[{"sin", "(", "x", ")"}]}], " ", "\[Equal]", " ", 
     RowBox[{"0", " ", "with", " ", 
      RowBox[{"x", "(", "0", ")"}]}]}], " ", "=", " ", 
    RowBox[{
     RowBox[{"x", "(", "1", ")"}], " ", "=", " ", "0"}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->603352701],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sols", " ", "=", " ", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"First", "[", 
       RowBox[{"NDSolve", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{
             RowBox[{
              RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
             RowBox[{"Sin", "[", 
              RowBox[{"x", "[", "t", "]"}], "]"}]}], " ", "\[Equal]", " ", 
            "0"}], ",", " ", 
           RowBox[{
            RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", 
            RowBox[{"x", "[", "10", "]"}], " ", "\[Equal]", " ", "0"}]}], 
          "}"}], ",", "x", ",", "t", ",", " ", 
         RowBox[{"Method", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<Shooting\>\"", ",", " ", 
            RowBox[{"\"\<StartingInitialConditions\>\"", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{
               RowBox[{
                RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], 
               ",", " ", 
               RowBox[{
                RowBox[{
                 RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", 
                "#"}]}], "}"}]}]}], "}"}]}]}], "]"}], "]"}], "&"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"1.5", ",", " ", "1.75", ",", " ", "2"}], "}"}]}], "]"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{
     RowBox[{"x", "[", "t", "]"}], " ", "/.", " ", "sols"}], "]"}], ",", " ", 
   
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Black", ",", " ", "Blue", ",", " ", "Green"}], "}"}]}]}], 
  "]"}]}], "Input",
 CellLabel->"In[105]:=",
 CellID->49231136],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[106]=",
 CellID->30064504]
}, Open  ]],

Cell["\<\
By default, shooting starts from the left side of the interval and shoots \
forward in time. There are cases where is it advantageous to go backwards, or \
even from a point somewhere in the middle of the interval.\
\>", "Text",
 CellID->163787919],

Cell["Consider the linear boundary value problem ", "Text",
 CellID->473868030],

Cell[TextData[{
 "                           ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "'''"}], 
      RowBox[{"(", "t", ")"}]}], " ", "-", " ", 
     RowBox[{"2", "\[Lambda]", " ", 
      RowBox[{"x", "''"}], 
      RowBox[{"(", "t", ")"}]}], " ", "-", " ", 
     RowBox[{
      SuperscriptBox["\[Lambda]", "2"], 
      RowBox[{"x", "'"}], 
      RowBox[{"(", "t", ")"}]}], " ", "+", " ", 
     RowBox[{"2", " ", 
      SuperscriptBox["\[Lambda]", "3"], 
      RowBox[{"x", "(", "t", ")"}]}]}], " ", "=", " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["\[Lambda]", "2"], "+", 
       SuperscriptBox["\[Pi]", "2"]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"2", " ", "\[Lambda]", " ", 
        RowBox[{"cos", "(", 
         RowBox[{"\[Pi]", " ", "t"}], ")"}]}], "+", 
       RowBox[{"\[Pi]", " ", 
        RowBox[{"sin", "(", 
         RowBox[{"\[Pi]", " ", "t"}], ")"}]}]}], ")"}]}]}], TraditionalForm]],
   "InlineMath"],
 "            \n                           ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"x", "(", "0", ")"}], " ", "=", " ", 
     RowBox[{"1", " ", "+", " ", 
      FractionBox[
       RowBox[{"1", "+", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{
          RowBox[{"-", "2"}], " ", "\[Lambda]"}]], "+", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"-", "\[Lambda]"}]]}], 
       RowBox[{"2", "+", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"-", "\[Lambda]"}]]}]]}]}], ",", " ", 
    RowBox[{
     RowBox[{"x", "(", "1", ")"}], "  ", "=", " ", "0"}], ",", " ", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "'"}], 
      RowBox[{"(", "1", ")"}]}], " ", "=", " ", 
     FractionBox[
      RowBox[{
       RowBox[{"3", " ", "\[Lambda]"}], "-", 
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"-", "\[Lambda]"}]], " ", "\[Lambda]"}]}], 
      RowBox[{"2", "+", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"-", "\[Lambda]"}]]}]]}]}], TraditionalForm]], "InlineMath"]
}], "Text",
 CellID->563834588],

Cell["that has a solution ", "Text",
 CellID->16284262],

Cell[BoxData[
 FormBox[
  RowBox[{"                                        ", 
   RowBox[{
    RowBox[{"x", "(", "t", ")"}], " ", "=", " ", 
    RowBox[{
     FractionBox[
      RowBox[{
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"\[Lambda]", " ", 
         RowBox[{"(", 
          RowBox[{"t", "-", "1"}], ")"}]}]], "+", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"2", " ", "\[Lambda]", " ", 
         RowBox[{"(", 
          RowBox[{"t", "-", "1"}], ")"}]}]], "+", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{
         RowBox[{"-", "\[Lambda]"}], " ", "t"}]]}], 
      RowBox[{"2", " ", "+", " ", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"-", "\[Lambda]"}]]}]], "+", 
     RowBox[{"cos", "(", 
      RowBox[{"\[Pi]", " ", "t"}], ")"}]}]}]}], TraditionalForm]], "Text",
 CellID->148768874],

Cell[TextData[{
 "For moderate values of \[Lambda], the initial value problem starting at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", " ", "=", " ", "0"}], TraditionalForm]], "InlineMath"],
 " becomes unstable because of the growing ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"\[Lambda]", " ", 
      RowBox[{"(", 
       RowBox[{"t", "-", "1"}], ")"}]}]], "and", "  ", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"2", " ", "\[Lambda]", " ", 
      RowBox[{"(", 
       RowBox[{"t", "-", "1"}], ")"}]}]]}], TraditionalForm]], "InlineMath"],
 " terms. Similarly, starting at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", " ", "=", " ", "1"}], TraditionalForm]], "InlineMath"],
 ", instability arises from the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", "\[Lambda]"}], " ", "t"}]], TraditionalForm]], 
  "InlineMath"],
 " term, though this is not as large as the term in the forward direction. \
Beyond some value of \[Lambda], shooting will not be able to get a good \
solution because the sensitivity in either direction will be too great.",
 " ",
 "However, up to that point, choosing a point in the interval that balances \
the growth in the two directions will give the best solution."
}], "Text",
 CellID->44686511],

Cell[TextData[{
 "This gives the equation, boundary conditions, and exact solution as ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " input."
}], "MathCaption",
 CellID->795575749],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eqn", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"x", "'''"}], "[", "t", "]"}], " ", "-", " ", 
     RowBox[{"2", "\[Lambda]", " ", 
      RowBox[{
       RowBox[{"x", "''"}], "[", "t", "]"}]}], " ", "-", " ", 
     RowBox[{
      SuperscriptBox["\[Lambda]", "2"], 
      RowBox[{
       RowBox[{"x", "'"}], "[", "t", "]"}]}], " ", "+", " ", 
     RowBox[{"2", 
      SuperscriptBox["\[Lambda]", "3"], 
      RowBox[{"x", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["\[Lambda]", "2"], "+", 
       SuperscriptBox["\[Pi]", "2"]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"2", " ", "\[Lambda]", " ", 
        RowBox[{"Cos", "[", 
         RowBox[{"\[Pi]", " ", "t"}], "]"}]}], "+", 
       RowBox[{"\[Pi]", " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"\[Pi]", " ", "t"}], "]"}]}]}], ")"}]}]}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"bcs", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", "  ", 
      RowBox[{"1", " ", "+", " ", 
       FractionBox[
        RowBox[{"1", "+", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{
           RowBox[{"-", "2"}], " ", "\[Lambda]"}]], "+", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "\[Lambda]"}]]}], 
        RowBox[{"2", "+", 
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "\[Lambda]"}]]}]]}]}], ",", " ", 
     RowBox[{
      RowBox[{"x", "[", "1", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "1", "]"}], " ", "\[Equal]", " ", 
      FractionBox[
       RowBox[{
        RowBox[{"3", " ", "\[Lambda]"}], "-", 
        RowBox[{
         SuperscriptBox["\[ExponentialE]", 
          RowBox[{"-", "\[Lambda]"}]], " ", "\[Lambda]"}]}], 
       RowBox[{"2", "+", 
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"-", "\[Lambda]"}]]}]]}]}], "}"}]}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{
   RowBox[{"xsol", "[", "t_", "]"}], " ", "=", "  ", 
   RowBox[{
    FractionBox[
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"\[Lambda]", " ", 
        RowBox[{"(", 
         RowBox[{"t", "-", "1"}], ")"}]}]], "+", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"2", " ", "\[Lambda]", " ", 
        RowBox[{"(", 
         RowBox[{"t", "-", "1"}], ")"}]}]], "+", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{"-", "\[Lambda]"}], " ", "t"}]]}], 
     RowBox[{"2", " ", "+", " ", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"-", "\[Lambda]"}]]}]], "+", 
    RowBox[{"Cos", "[", 
     RowBox[{"\[Pi]", " ", "t"}], "]"}]}]}], ";"}]}], "Input",
 CellLabel->"In[107]:=",
 CellID->232407840],

Cell[TextData[{
 "This solves the system with \[Lambda] = 10 shooting from the default ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", " ", "=", " ", "0"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->7358722],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\[Lambda]", " ", "=", " ", "10"}], "}"}], ",", " ", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"sol", " ", "=", " ", 
     RowBox[{"First", "[", 
      RowBox[{"NDSolve", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"eqn", ",", " ", "bcs"}], "}"}], ",", " ", "x", ",", " ", 
        "t"}], "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"xsol", "[", "t", "]"}], ",", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], " ", "/.", " ", "sol"}]}], "}"}], ",", 
      " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}]}], "]"}]], "Input",\

 CellLabel->"In[110]:=",
 CellID->183247138],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"bvluc\"\>"}], ":", 
  " ", "\<\"The equations derived from the boundary conditions are \
numerically ill-conditioned. The boundary conditions may not be sufficient to \
uniquely define a solution. The computed solution may match the boundary \
conditions poorly.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[110]:=",
 CellID->32853519],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"berr\"\>"}], ":", 
  " ", "\<\"There are significant errors \
\\!\\({\\(\\(-1.1102230246251565`*^-16\\)\\), 6.951230764569528`*^-6, \
0.00013902894030870527`}\\) in the boundary value residuals.  Returning the \
best solution found.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[110]:=",
 CellID->733380702],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[110]=",
 CellID->610290743]
}, Open  ]],

Cell[TextData[{
 "Shooting from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", " ", "=", " ", "0"}], TraditionalForm]], "InlineMath"],
 ", the shooting method gives warnings about an ill conditioned matrix, and \
further that the boundary conditions are not satisfied as well as they should \
be. This is because a small error at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", " ", "=", " ", "0"}], TraditionalForm]], "InlineMath"],
 " is amplified by ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     RowBox[{
      SuperscriptBox["e", "20"], "\[TildeEqual]"}],
     TraditionalForm], 
    RowBox[{
     RowBox[{"4", "\[Cross]", 
      SuperscriptBox["10", "8"]}], "."}]}], TraditionalForm]], "InlineMath"],
 " Since the reciprocal of this is of the same order of magnitude as the \
local truncation error, visible errors as those seen in the plot are not \
surprising.",
 " ",
 "In the reverse direction, the magnification will be much less: ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     RowBox[{
      SuperscriptBox["e", "10"], "\[TildeEqual]"}],
     TraditionalForm], 
    RowBox[{"2", "\[Cross]", 
     SuperscriptBox["10", "4"]}]}], TraditionalForm]], "InlineMath"],
 ", so the solution should be much better."
}], "Text",
 CellID->120337663],

Cell[TextData[{
 "This computes the solution using shooting from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "=", " ", "1"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->4681285],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\[Lambda]", " ", "=", " ", "10"}], "}"}], ",", " ", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"sol", " ", "=", " ", 
     RowBox[{"First", "[", 
      RowBox[{"NDSolve", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"eqn", ",", " ", "bcs"}], "}"}], ",", " ", "x", ",", " ", 
        "t", ",", " ", "\[IndentingNewLine]", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<Shooting\>\"", ",", " ", 
           RowBox[{"\"\<StartingInitialConditions\>\"", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{
               RowBox[{"x", "[", "1", "]"}], " ", "\[Equal]", " ", "0"}], ",",
               " ", 
              RowBox[{
               RowBox[{
                RowBox[{"x", "'"}], "[", "1", "]"}], " ", "\[Equal]", " ", 
               FractionBox[
                RowBox[{
                 RowBox[{"3", " ", "\[Lambda]"}], "-", 
                 RowBox[{
                  SuperscriptBox["\[ExponentialE]", 
                   RowBox[{"-", "\[Lambda]"}]], " ", "\[Lambda]"}]}], 
                RowBox[{"2", "+", 
                 SuperscriptBox["\[ExponentialE]", 
                  RowBox[{"-", "\[Lambda]"}]]}]]}], ",", " ", 
              RowBox[{
               RowBox[{
                RowBox[{"x", "''"}], "[", "1", "]"}], " ", "\[Equal]", " ", 
               "0"}]}], "}"}]}]}], "}"}]}]}], "]"}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"xsol", "[", "t", "]"}], ",", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], " ", "/.", " ", "sol"}]}], "}"}], ",", 
      " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}]}], "]"}]], "Input",\

 CellLabel->"In[111]:=",
 CellID->65999615],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[111]=",
 CellID->5245260]
}, Open  ]],

Cell[TextData[{
 "A good point to choose is actually one that will balance the sensitivity in \
each direction, which is about at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", " ", "=", " ", 
    RowBox[{"2", "/", "3"}]}], TraditionalForm]], "InlineMath"],
 ".",
 " ",
 "With this, the error with \[Lambda] = 15 will still be under reasonable \
control."
}], "Text",
 CellID->97834841],

Cell[TextData[{
 "This computes the solution for \[Lambda] = 15 shooting from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", " ", "=", " ", 
    RowBox[{"2", "/", "3."}]}], TraditionalForm]], "InlineMath"]
}], "MathCaption",
 CellID->47675345],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\[Lambda]", " ", "=", " ", "15"}], "}"}], ",", " ", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"sol", " ", "=", " ", 
     RowBox[{"First", "[", 
      RowBox[{"NDSolve", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"eqn", ",", " ", "bcs"}], "}"}], ",", " ", "x", ",", " ", 
        "t", ",", " ", "\[IndentingNewLine]", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<Shooting\>\"", ",", " ", 
           RowBox[{"\"\<StartingInitialConditions\>\"", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{
              RowBox[{
               RowBox[{"x", "[", 
                RowBox[{"2", "/", "3"}], "]"}], " ", "\[Equal]", " ", "0"}], 
              ",", " ", 
              RowBox[{
               RowBox[{
                RowBox[{"x", "'"}], "[", 
                RowBox[{"2", "/", "3"}], "]"}], " ", "\[Equal]", " ", "0"}], 
              ",", " ", 
              RowBox[{
               RowBox[{
                RowBox[{"x", "''"}], "[", 
                RowBox[{"2", "/", "3"}], "]"}], " ", "\[Equal]", " ", "0"}]}],
              "}"}]}]}], "}"}]}]}], "]"}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"Plot", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"xsol", "[", "t", "]"}], ",", 
        RowBox[{
         RowBox[{"x", "[", "t", "]"}], " ", "/.", " ", "sol"}]}], "}"}], ",", 
      " ", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}]}], "]"}]], "Input",\

 CellLabel->"In[112]:=",
 CellID->54666566],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 101},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[112]=",
 CellID->171794514]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Subsection",
 CellTags->"s:5",
 CellID->519011742],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<StartingInitialConditions\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "The initial conditions to initiate the shooting method from.", 
     "TableText"]},
   {"\"\<ImplicitSolver\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "The method to use for solving the implicit equation defined by the \
boundary conditions. This should be an acceptable value for the ",
     ButtonBox["Method",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Method"],
     " option of ",
     ButtonBox["FindRoot",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/FindRoot"],
     "."
    }], "TableText"]},
   {"\"\<MaxIterations\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "How many iterations to use for the implicit solver method.", 
     "TableText"]},
   {"\"\<Method\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "The method to use for integrating the system of ODEs.", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->508999769],

Cell["Shooting method options", "Caption",
 CellID->18856673]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Chasing Method", "Section",
 CellID->948663323],

Cell[TextData[{
 "The method of chasing came from a manuscript of Gel'fand and Lokutsiyevskii \
first published in English in [",
 ButtonBox["BZ65",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#5356"],
 "] and further described in [",
 ButtonBox["Na79",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#712"],
 "]. The idea is to establish a set of auxiliary problems that can be solved \
to find initial conditions at one of the boundaries. Once the initial \
conditions are determined, the usual methods for solving initial value \
problems can be applied.",
 " ",
 "The chasing method is, in effect, a shooting method that uses the linearity \
of the problem to good advantage."
}], "Text",
 CellID->262954798],

Cell["Consider the linear ODE", "Text",
 CellID->847545622],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SuperscriptBox["X", "\[Prime]",
     MultilineFunction->None], " ", 
    RowBox[{"(", "t", ")"}]}], "==", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"A", "(", "t", ")"}], " ", 
     RowBox[{"X", "(", "t", ")"}]}], " ", "+", " ", 
    RowBox[{
     SubscriptBox["A", "0"], "(", "t", ")"}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellID->119919937],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"X", "(", "t", ")"}], " ", "=", " ", "("}], TraditionalForm]], 
  "InlineMath"],
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], "(", "t", ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "2"], "(", "t", ")"}], TraditionalForm]], 
  "InlineMath"],
 ", \[CenterEllipsis], ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "n"], "(", "t", ")"}], TraditionalForm]], 
  "InlineMath"],
 "), ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"A", "(", "t", ")"}], " "}], TraditionalForm]], "InlineMath"],
 "is the coefficient matrix, and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["A", "0"], "(", "t", ")"}], " "}], TraditionalForm]], 
  "InlineMath"],
 "is the inhomogeneous coefficient vector, with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", " "}], TraditionalForm]], "InlineMath"],
 "linear boundary conditions "
}], "Text",
 CellID->400908072],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["B", "i"], " ", ".", " ", 
     RowBox[{"X", "(", 
      SubscriptBox["t", "i"], ")"}]}], " ", "\[Equal]", 
    SubscriptBox["b", "i0"]}], ",", " ", 
   RowBox[{"i", " ", "=", " ", "1"}], ",", " ", "2", ",", " ", 
   "\[CenterEllipsis]", ",", " ", "n"}], TraditionalForm]], "NumberedEquation",\

 CellID->947916108],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["B", "i"], " ", "=", " ", 
    RowBox[{"(", 
     RowBox[{
      SubscriptBox["b", 
       StyleBox[
        RowBox[{
         StyleBox["i",
          FontSlant->"Italic"], "1"}]]], ",", 
      SubscriptBox["b", 
       StyleBox[
        RowBox[{
         StyleBox["i",
          FontSlant->"Italic"], "2"}]]], ",", "\[CenterEllipsis]", ",", 
      FormBox[
       SubscriptBox["b", 
        StyleBox["in",
         FontSlant->"Italic"]],
       TraditionalForm]}], ")"}]}], TraditionalForm]], "InlineMath"],
 "is a coefficient vector.",
 " ",
 "From this, construct the augmented homogenous system"
}], "Text",
 CellID->887862838],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox[
      OverscriptBox["X", "_"], "\[Prime]",
      MultilineFunction->None], " ", 
     RowBox[{"(", "t", ")"}]}], "=", " ", 
    RowBox[{
     RowBox[{
      OverscriptBox["A", "_"], "(", "t", ")"}], " ", 
     RowBox[{
      OverscriptBox["X", "_"], "(", "t", ")"}]}]}], ",", "         ", 
   RowBox[{
    RowBox[{
     SubscriptBox[
      OverscriptBox["B", "_"], "i"], " ", ".", " ", 
     RowBox[{
      OverscriptBox["X", "_"], "(", 
      SubscriptBox["t", "i"], ")"}]}], " ", "=", " ", "0"}]}], 
  TraditionalForm]], "NumberedEquation",
 CellID->21985348],

Cell["where", "Text",
 CellID->1301566606],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     OverscriptBox["X", "_"], "(", "t", ")"}], " ", "=", " ", 
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"1"},
       {
        RowBox[{
         SubscriptBox["x", "1"], "(", "t", ")"}]},
       {
        RowBox[{
         SubscriptBox["x", "2"], "(", "t", ")"}]},
       {"\[VerticalEllipsis]"},
       {
        RowBox[{
         SubscriptBox["x", "n"], "(", "t", ")"}]}
      }], "\[NoBreak]", ")"}]}], "  ", ",", "   ", 
   RowBox[{
    RowBox[{
     OverscriptBox["A", "_"], "(", "t", ")"}], " ", "=", " ", 
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        RowBox[{
         SubscriptBox["a", "01"], "(", "t", ")"}], 
        RowBox[{
         SubscriptBox["a", "11"], "(", "t", ")"}], 
        RowBox[{
         SubscriptBox["a", "12"], "(", "t", ")"}], "\[CenterEllipsis]", 
        RowBox[{
         SubscriptBox["a", 
          RowBox[{"1", " ", "n"}]], "(", "t", ")"}]},
       {
        RowBox[{
         SubscriptBox["a", "02"], "(", "t", ")"}], 
        RowBox[{
         SubscriptBox["a", "21"], "(", "t", ")"}], 
        RowBox[{
         SubscriptBox["a", "22"], "(", "t", ")"}], "\[CenterEllipsis]", 
        RowBox[{
         SubscriptBox["a", 
          RowBox[{"2", " ", "n"}]], "(", "t", ")"}]},
       {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]", 
        "\[DescendingEllipsis]", "\[VerticalEllipsis]"},
       {
        RowBox[{
         SubscriptBox["a", 
          RowBox[{"0", " ", "n"}]], "(", "t", ")"}], 
        RowBox[{
         SubscriptBox["a", "n1"], "(", "t", ")"}], 
        RowBox[{
         SubscriptBox["a", "n2"], "(", "t", ")"}], "\[CenterEllipsis]", 
        RowBox[{
         SubscriptBox["a", "nn"], "(", "t", ")"}]},
       {"0", "0", "0", "\[CenterEllipsis]", "0"}
      }], "\[NoBreak]", ")"}]}], ",", " ", 
   RowBox[{
    RowBox[{"and", "  ", 
     SubscriptBox[
      OverscriptBox["B", "_"], "i"]}], " ", "=", "  ", 
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {
        SubscriptBox["b", 
         StyleBox[
          RowBox[{
           StyleBox["i",
            FontSlant->"Italic"], "0"}]]]},
       {
        SubscriptBox["b", 
         StyleBox[
          RowBox[{
           StyleBox["i",
            FontSlant->"Italic"], "1"}]]]},
       {
        SubscriptBox["b", 
         StyleBox[
          RowBox[{
           StyleBox["i",
            FontSlant->"Italic"], "2"}]]]},
       {"\[VerticalEllipsis]"},
       {
        SubscriptBox["b", 
         StyleBox["in",
          FontSlant->"Italic"]]}
      }], "\[NoBreak]", ")"}]}]}], TraditionalForm]], "DisplayMath",
 CellID->2004286477],

Cell[TextData[{
 "The chasing method amounts to finding a vector function ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}], TraditionalForm]], 
  "InlineMath"],
 "such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalPhi]", "i"], "(", 
    SubscriptBox["t", "i"], ")"}], TraditionalForm]], "InlineMath"],
 " = ",
 Cell[BoxData[
  FormBox[
   OverscriptBox[
    SubscriptBox["B", "i"], "_"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}], " ", 
     RowBox[{
      OverscriptBox["X", "_"], "(", "t", ")"}]}], "=", "0"}], 
   TraditionalForm]], "InlineMath"],
 ". Once the function ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}], " "}], 
   TraditionalForm]], "InlineMath"],
 "is known, if there is a full set of boundary conditions, solving "
}], "Text",
 CellID->1029232038],

Cell[BoxData[
 FormBox[
  RowBox[{" ", 
   RowBox[{
    RowBox[{
     RowBox[{"(", "\[NoBreak]", GridBox[{
        {
         RowBox[{
          SubscriptBox["\[CapitalPhi]", "1"], "(", 
          SubscriptBox["t", "0"], ")"}]},
        {
         RowBox[{
          SubscriptBox["\[CapitalPhi]", "2"], "(", 
          SubscriptBox["t", "0"], ")"}]},
        {"\[VerticalEllipsis]"},
        {
         RowBox[{
          SubscriptBox["\[CapitalPhi]", "n"], "(", 
          SubscriptBox["t", "0"], ")"}]}
       }], "\[NoBreak]", ")"}], " ", 
     RowBox[{
      OverscriptBox["X", "_"], "(", 
      SubscriptBox["t", "0"], ")"}]}], " ", "=", " ", "0"}]}], 
  TraditionalForm]], "NumberedEquation",
 CellID->205611458],

Cell[TextData[{
 "can be used to determine initial conditions ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      SubscriptBox["x", "1"], "(", 
      SubscriptBox["t", "0"], ")"}], ",", " ", 
     RowBox[{
      SubscriptBox["x", "2"], "(", 
      SubscriptBox["t", "0"], ")"}], ",", " ", "\[CenterEllipsis]", " ", ",", 
     " ", 
     RowBox[{
      SubscriptBox["x", "n"], "(", 
      SubscriptBox["t", "0"], ")"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " that can be used with the usual initial value problem solvers. Note that \
the solution to system (3) is nontrivial because the first component of ",
 Cell[BoxData[
  FormBox["X", TraditionalForm]], "InlineMath"],
 "is always 1. "
}], "Text",
 CellID->1228512178],

Cell[TextData[{
 "Thus, solving the boundary value problem is reduced to solving the \
auxiliary problems for the ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}], TraditionalForm]], 
  "InlineMath"],
 ".",
 " ",
 "Differentiating the equation for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalPhi]", "i"], " ", 
    RowBox[{"(", "t", ")"}]}], TraditionalForm]], "InlineMath"],
 "gives"
}], "Text",
 CellID->942597820],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}], " ", 
     RowBox[{
      SuperscriptBox[
       OverscriptBox["X", "_"], "\[Prime]",
       MultilineFunction->None], "(", "t", ")"}]}], "+", 
    RowBox[{
     RowBox[{
      OverscriptBox["X", "_"], "(", "t", ")"}], " ", 
     RowBox[{
      SuperscriptBox[
       SubscriptBox["\[CapitalPhi]", "i"], "\[Prime]",
       MultilineFunction->None], "(", "t", ")"}]}]}], " ", "=", " ", "0"}], 
  TraditionalForm]], "DisplayMath",
 CellID->28632706],

Cell["Substituting the differential equation, ", "Text",
 CellID->321043162],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{
      OverscriptBox["A", "_"], "(", "t", ")"}], " ", 
     RowBox[{
      OverscriptBox["X", "_"], "(", "t", ")"}], " ", 
     RowBox[{
      SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}]}], "+", 
    RowBox[{
     RowBox[{
      OverscriptBox["X", "_"], "(", "t", ")"}], " ", 
     RowBox[{
      SuperscriptBox[
       SubscriptBox["\[CapitalPhi]", "i"], "\[Prime]",
       MultilineFunction->None], "(", "t", ")"}]}]}], " ", "=", " ", "0"}], 
  TraditionalForm]], "DisplayMath",
 CellID->1348388825],

Cell["and transposing", "Text",
 CellID->1283149602],

Cell[BoxData[
 FormBox[
  RowBox[{" ", 
   RowBox[{
    RowBox[{
     RowBox[{
      OverscriptBox["X", "_"], "(", "t", ")"}], 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["\[CapitalPhi]", "i"], "\[Prime]",
         MultilineFunction->None], "(", "t", ")"}], " ", "+", 
       RowBox[{
        RowBox[{
         SuperscriptBox[
          OverscriptBox["A", "_"], "T"], "(", "t", ")"}], " ", 
        RowBox[{
         SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}]}]}], ")"}]}], 
    " ", "=", " ", "0"}]}], TraditionalForm]], "DisplayMath",
 CellID->2093107450],

Cell[TextData[{
 "Since this should hold for all solutions ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["X", "_"], TraditionalForm]], "InlineMath"],
 ", we have the initial value problem for ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "i"], TraditionalForm]], "InlineMath"],
 ","
}], "Text",
 CellID->1398185347],

Cell[BoxData[
 FormBox[
  RowBox[{" ", 
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox[
       SubscriptBox["\[CapitalPhi]", "i"], "\[Prime]",
       MultilineFunction->None], "(", "t", ")"}], " ", "+", 
     RowBox[{
      RowBox[{
       SuperscriptBox[
        OverscriptBox["A", "_"], "T"], "(", "t", ")"}], " ", 
      RowBox[{
       SubscriptBox["\[CapitalPhi]", "i"], "(", "t", ")"}]}]}], "=", " ", 
    RowBox[{
     RowBox[{"0", " ", "with", " ", "initial", " ", "condition", "  ", 
      RowBox[{
       SubscriptBox["\[CapitalPhi]", "i"], "(", 
       SubscriptBox["t", "i"], ")"}]}], " ", "=", " ", 
     SubscriptBox["B", "i"]}]}]}], TraditionalForm]], "NumberedEquation",
 CellID->99789938],

Cell[TextData[{
 "Given ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["t", "0"], TraditionalForm]], "InlineMath"],
 "where we want to have solutions to all of the boundary value problems, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " just uses ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " to solve the auxiliary problems for ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "1"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "2"], TraditionalForm]], "InlineMath"],
 ", ... ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "m"], TraditionalForm]], "InlineMath"],
 " by integrating them to ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["t", "0"], TraditionalForm]], "InlineMath"],
 ". The results are then combined into the matrix of Equation (3) that is \
solved for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"X", "(", 
    SubscriptBox["t", "0"], ")"}], TraditionalForm]], "InlineMath"],
 " to obtain the initial value problem that ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " integrates to give the returned solution. "
}], "Text",
 CellID->93361577],

Cell[TextData[{
 "This variant of the method is further described in and used by the ",
 StyleBox["MathSource",
  FontSlant->"Italic"],
 " package [",
 ButtonBox["R98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#20165"],
 "], which also allows you to solve linear eigenvalue problems."
}], "Text",
 CellID->1076859350],

Cell[TextData[{
 "There is an alternative, nonlinear way to set up the auxiliary problems \
that is closer to the original method proposed by Gel'fand and \
Lokutsiyevskii. Assume that the boundary conditions are linearly independent \
(if not, then the problem is insufficiently specified). Then in each ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["B", "i"], TraditionalForm]], "InlineMath"],
 ", there is at least one nonzero component. Without loss of generality, \
assume that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["b", 
     StyleBox["ij",
      FontSlant->"Italic"]], "\[NotEqual]", " ", "0."}], TraditionalForm]], 
  "InlineMath"],
 " Now solve for ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", 
    StyleBox["ij",
     FontSlant->"Italic"]], TraditionalForm]], "InlineMath"],
 "in terms of the other components of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "i"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", "ij"], TraditionalForm]], "InlineMath"],
 " = ",
 Cell[BoxData[
  FormBox[
   OverscriptBox[
    SubscriptBox["B", "i"], "~"], TraditionalForm]], "InlineMath"],
 ".",
 Cell[BoxData[
  FormBox[
   OverscriptBox[
    FormBox[
     SubscriptBox["\[CapitalPhi]", "i"],
     TraditionalForm], "~"], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   OverscriptBox[
    FormBox[
     SubscriptBox["\[CapitalPhi]", "i"],
     TraditionalForm], "~"], TraditionalForm]], "InlineMath"],
 " = (1, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", 
    StyleBox[
     RowBox[{
      StyleBox["i",
       FontSlant->"Italic"], "1"}]]], TraditionalForm]], "InlineMath"],
 ",\[CenterEllipsis],",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", 
    RowBox[{
     StyleBox["ij",
      FontSlant->"Italic"], "-", "1"}]], TraditionalForm]], "InlineMath"],
 ",\[CenterEllipsis],",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", 
    RowBox[{
     StyleBox["ij",
      FontSlant->"Italic"], "+", "1"}]], TraditionalForm]], "InlineMath"],
 ",\[CenterEllipsis],",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", 
    StyleBox["in",
     FontSlant->"Italic"]], TraditionalForm]], "InlineMath"],
 ") and ",
 Cell[BoxData[
  FormBox[
   OverscriptBox[
    SubscriptBox["B", "i"], "~"], TraditionalForm]], "InlineMath"],
 " = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       FormBox[
        RowBox[{"(", 
         SubscriptBox["b", 
          RowBox[{
           StyleBox[
            RowBox[{
             StyleBox["i",
              FontSlant->"Italic"], "0"}]], ",", " "}]]}],
        TraditionalForm], 
       FormBox[
        SubscriptBox["b", 
         StyleBox[
          RowBox[{
           StyleBox["i",
            FontSlant->"Italic"], "1"}]]],
        TraditionalForm]}], ",", "\[CenterEllipsis]", ",", 
      FormBox[
       SubscriptBox["b", 
        RowBox[{
         StyleBox["ij",
          FontSlant->"Italic"], "-", "1"}]],
       TraditionalForm], ",", "\[CenterEllipsis]", ",", 
      FormBox[
       SubscriptBox["b", 
        RowBox[{
         StyleBox["ij",
          FontSlant->"Italic"], "+", "1"}]],
       TraditionalForm], ",", "\[CenterEllipsis]", ",", 
      FormBox[
       SubscriptBox["b", 
        StyleBox["in",
         FontSlant->"Italic"]],
       TraditionalForm]}], ")"}], "/", 
    RowBox[{"-", 
     SubscriptBox["b", 
      StyleBox["ij",
       FontSlant->"Italic"]]}]}], TraditionalForm]], "InlineMath"],
 ". Using (5) and replacing ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalPhi]", 
    StyleBox["ij",
     FontSlant->"Italic"]], TraditionalForm]], "InlineMath"],
 ", and thinking of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "n"], "(", "t", ")"}], TraditionalForm]], 
  "InlineMath"],
 "in terms of the other components of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
 "we get the nonlinear equation"
}], "Text",
 CellID->431346174],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SuperscriptBox[
     OverscriptBox[
      SubscriptBox["\[CapitalPhi]", "i"], "~"], "\[Prime]",
     MultilineFunction->None], "(", "t", ")"}], "=", 
   RowBox[{
    RowBox[{
     RowBox[{"-", 
      RowBox[{
       SuperscriptBox[
        OverscriptBox["A", "~"], "T"], "[", "t", "]"}]}], " ", 
     RowBox[{
      OverscriptBox[
       SubscriptBox["\[CapitalPhi]", "i"], "~"], "(", "t", ")"}]}], "+", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["A", "j"], ".", 
       RowBox[{
        OverscriptBox[
         SubscriptBox["\[CapitalPhi]", "i"], "~"], "(", "t", ")"}]}], ")"}], 
     " ", 
     RowBox[{
      OverscriptBox[
       SubscriptBox["\[CapitalPhi]", "i"], "~"], "(", "t", ")"}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->2074262724],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["A", "~"], TraditionalForm]], "InlineMath"],
 " is ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 " with the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["j", "th"], TraditionalForm]]],
 "column removed and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["A", "j"], TraditionalForm]], "InlineMath"],
 "is the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["j", 
    RowBox[{"th", " "}]], TraditionalForm]], "InlineMath"],
 "column of A. The nonlinear method can be more numerically stable than the \
linear method, but it has the disadvantage that integration along the real \
line may lead to singularities. This problem can be eliminated by integrating \
on a contour in the complex plane. However, the integration in the complex \
plane typically has more numerical error than a simple integration along the \
real line, so in practice, the nonlinear method does not typically give \
results better than the linear method. For this reason, and because it is \
also generally faster, the default for ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " is to use the linear method."
}], "Text",
 CellID->868453947],

Cell["\<\
This solves a two-point boundary value problem for a second-order equation.\
\>", "MathCaption",
 CellID->1224517775],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"nsol1", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{
         RowBox[{"y", "[", "t", "]"}], "/", "4"}]}], " ", "\[Equal]", " ", 
       "8"}], ",", " ", 
      RowBox[{
       RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
      RowBox[{
       RowBox[{"y", "[", "10", "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}], 
    ",", "y", ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[113]:=",
 CellID->786868787],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"y", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "10.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{316, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[113]=",
 CellID->4706927]
}, Open  ]],

Cell["This shows a plot of the solution.", "MathCaption",
 CellID->690415336],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"First", "[", 
    RowBox[{
     RowBox[{"y", "[", "t", "]"}], " ", "/.", " ", "nsol1"}], "]"}], ",", " ",
    
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[114]:=",
 CellID->1697043452],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 230},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[114]=",
 CellID->582922320]
}, Open  ]],

Cell[TextData[{
 "The solver can solve multipoint boundary value problems of linear systems \
of equations. (Note that each boundary equation must be at one specific value \
of ",
 StyleBox["t",
  FontSlant->"Italic"],
 ".)"
}], "MathCaption",
 CellID->953370135],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"bconds", " ", "=", " ", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "[", "0", "]"}], " ", "+", " ", 
       RowBox[{
        RowBox[{"x", "'"}], "[", "0", "]"}], " ", "+", " ", 
       RowBox[{"y", "[", "0", "]"}], " ", "+", " ", 
       RowBox[{
        RowBox[{"y", "'"}], "[", "0", "]"}]}], " ", "\[Equal]", " ", "1"}], 
     ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "[", "1", "]"}], " ", "+", " ", 
       RowBox[{"2", " ", 
        RowBox[{
         RowBox[{"x", "'"}], "[", "1", "]"}]}], " ", "+", " ", 
       RowBox[{"3", " ", 
        RowBox[{"y", "[", "1", "]"}]}], " ", "+", " ", 
       RowBox[{"4", 
        RowBox[{
         RowBox[{"y", "'"}], "[", "1", "]"}]}]}], " ", "\[Equal]", " ", "5"}],
      ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "[", "2", "]"}], " ", "+", " ", 
       RowBox[{"2", " ", 
        RowBox[{
         RowBox[{"y", "'"}], "[", "2", "]"}]}]}], " ", "\[Equal]", " ", "4"}],
      ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"x", "[", "3", "]"}], " ", "-", " ", 
       RowBox[{
        RowBox[{"x", "'"}], "[", "3", "]"}]}], " ", "\[Equal]", " ", "7"}]}], 
    "}"}]}], ";"}], "\n", 
 RowBox[{"nsol2", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{
    RowBox[{"{", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{"x", "[", "t", "]"}], " ", "+", " ", 
        RowBox[{"y", "[", "t", "]"}]}], " ", "\[Equal]", " ", "t"}], ",", 
      RowBox[{
       RowBox[{
        RowBox[{
         RowBox[{"y", "''"}], "[", "t", "]"}], " ", "+", " ", 
        RowBox[{"y", "[", "t", "]"}]}], " ", "\[Equal]", " ", 
       RowBox[{"Cos", "[", "t", "]"}]}], ",", "\[IndentingNewLine]", 
      "bconds"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{"x", ",", "y"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "4"}], "}"}]}], "]"}]}]}], "Input",
 CellLabel->"In[115]:=",
 CellID->397105427],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "4.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}], ",", 
    RowBox[{"y", "\[Rule]", 
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "4.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False]}]}], "}"}], "}"}]], "Output",
 ImageSize->{309, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[116]=",
 CellID->58253176]
}, Open  ]],

Cell[TextData[{
 "In general, you cannot expect the boundary value equations to be satisfied \
to the close tolerance of ",
 StyleBox["Equal", "MR"],
 "."
}], "Text",
 CellID->1354278207],

Cell["\<\
This checks to see if the boundary conditions are \"satisfied\".\
\>", "MathCaption",
 CellID->1345021535],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"bconds", " ", "/.", " ", 
  RowBox[{"First", "[", "nsol2", "]"}]}]], "Input",
 CellLabel->"In[117]:=",
 CellID->407573496],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"True", ",", "False", ",", "False", ",", "False"}], "}"}]], "Output",\

 ImageSize->{185, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[117]=",
 CellID->454740148]
}, Open  ]],

Cell[TextData[{
 "They are typically only be satisfied at most tolerances that come from the \
",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " options of ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ". Usually, the actual accuracy and precision is less because these are used \
for local, not global, error control."
}], "Text",
 CellID->499104493],

Cell["\<\
This checks the residual error at each of the boundary conditions.\
\>", "MathCaption",
 CellID->264376325],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Apply", "[", 
   RowBox[{"Subtract", ",", " ", "bconds", ",", " ", "1"}], "]"}], " ", "/.", 
  " ", 
  RowBox[{"First", "[", "nsol2", "]"}]}]], "Input",
 CellLabel->"In[118]:=",
 CellID->258011946],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.`", ",", 
   RowBox[{"-", "2.5751019272490794`*^-7"}], ",", 
   RowBox[{"-", "4.1335717959611884`*^-8"}], ",", 
   RowBox[{"-", "2.9550824010016186`*^-8"}]}], "}"}]], "Output",
 ImageSize->{336, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[118]=",
 CellID->31018780]
}, Open  ]],

Cell[TextData[{
 "When you give ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " a problem that has no solution, numerical error may make it appear to be a \
solvable problem. Typically, ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " will issue a warning message."
}], "Text",
 CellID->1492888503],

Cell["This is a boundary value problem that has no solution.", "MathCaption",
 CellID->1200609766],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
       RowBox[{"x", "[", "t", "]"}]}], " ", "\[Equal]", " ", "0"}], ",", " ", 
     
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
     RowBox[{
      RowBox[{"x", "[", "Pi", "]"}], " ", "\[Equal]", " ", "0"}]}], "}"}], 
   ",", "x", ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "Pi"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<Chasing\>\""}]}], "]"}]], "Input",
 CellLabel->"In[125]:=",
 CellID->733348657],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"bvluc\"\>"}], ":", 
  " ", "\<\"The equations derived from the boundary conditions are \
numerically ill-conditioned. The boundary conditions may not be sufficient to \
uniquely define a solution. The computed solution may match the boundary \
conditions poorly.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[125]:=",
 CellID->87486231],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"x", "\[Rule]", 
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "3.141592653589793`"}], "}"}], "}"}], 
       ",", "\<\"<>\"\>"}], "]"}],
     False,
     Editable->False]}], "}"}], "}"}]], "Output",
 ImageSize->{344, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[125]=",
 CellID->645779371]
}, Open  ]],

Cell["\<\
In this case, it is not able to integrate over the entire interval because of \
nonexistence.\
\>", "Text",
 CellID->2016498260],

Cell["\<\
Another situation in which the equations can be ill-conditioned is when the \
boundary conditions do not give a unique solution.\
\>", "Text",
 CellID->1759555467],

Cell[TextData[{
 "Here is a boundary value problem that does not have a unique solution. Its \
general solution is shown as computed symbolically with ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->1799700291],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dsol", " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{"x", " ", "/.", " ", 
    RowBox[{"DSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
          RowBox[{"x", "[", "t", "]"}]}], " ", "\[Equal]", " ", "t"}], ",", 
        " ", 
        RowBox[{
         RowBox[{
          RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], 
        ",", " ", 
        RowBox[{
         RowBox[{"x", "[", 
          RowBox[{"Pi", "/", "2"}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{"Pi", "/", "2"}]}]}], "}"}], ",", "x", ",", "t"}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[120]:=",
 CellID->2116288198],

Cell[BoxData[
 RowBox[{
  RowBox[{"DSolve", "::", "\<\"bvsing\"\>"}], ":", 
  " ", "\<\"Unable to resolve some of the arbitrary constants in the general \
solution using the given boundary conditions. It is possible that some of the \
conditions have been specified at a singular point for the equation.\"\>"}]], \
"Message", "MSG",
 CellLabel->"During evaluation of In[120]:=",
 CellID->332611952],

Cell[BoxData[
 RowBox[{"Function", "[", 
  RowBox[{
   RowBox[{"{", "t", "}"}], ",", 
   RowBox[{"t", "+", 
    RowBox[{
     RowBox[{"C", "[", "1", "]"}], " ", 
     RowBox[{"Cos", "[", "t", "]"}]}]}]}], "]"}]], "Output",
 ImageSize->{199, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[120]=",
 CellID->149070785]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " issues a warning message because the matrix to solve for the initial \
conditions is singular, but has a solution."
}], "MathCaption",
 CellID->400150724],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"onesol", " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{"x", " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
          RowBox[{"x", "[", "t", "]"}]}], " ", "\[Equal]", " ", "t"}], ",", 
        " ", 
        RowBox[{
         RowBox[{
          RowBox[{"x", "'"}], "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], 
        ",", " ", 
        RowBox[{
         RowBox[{"x", "[", 
          RowBox[{"Pi", "/", "2"}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{"Pi", "/", "2"}]}]}], "}"}], ",", "x", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", " ", "0", ",", " ", 
        RowBox[{"Pi", "/", "2"}]}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", "\"\<Chasing\>\""}]}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[122]:=",
 CellID->1779696120],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"bvluc\"\>"}], ":", 
  " ", "\<\"The equations derived from the boundary conditions are \
numerically ill-conditioned. The boundary conditions may not be sufficient to \
uniquely define a solution. The computed solution may match the boundary \
conditions poorly.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[122]:=",
 CellID->456919939],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0.`", ",", "1.5707963267948966`"}], "}"}], "}"}], 
    ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{286, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[122]=",
 CellID->466855944]
}, Open  ]],

Cell["\<\
You can identify which solution it found by fitting it to the interpolating \
points. This makes a plot of the error relative to the actual best fit \
solution.\
\>", "MathCaption",
 CellID->1624668500],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"ip", " ", "=", " ", 
   RowBox[{"onesol", "@", 
    RowBox[{"\"\<Coordinates\>\"", "[", "1", "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"points", " ", "=", " ", 
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"ip", ",", " ", 
      RowBox[{"onesol", "[", "ip", "]"}]}], "}"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"model", " ", "=", " ", 
   RowBox[{
    RowBox[{"dsol", "[", "t", "]"}], " ", "/.", " ", 
    RowBox[{
     RowBox[{"C", "[", "1", "]"}], "\[Rule]", "\[Alpha]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"fit", " ", "=", " ", 
   RowBox[{"FindFit", "[", 
    RowBox[{"points", ",", " ", "model", ",", "\[Alpha]", ",", " ", "t"}], 
    "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"ip", ",", " ", 
     RowBox[{
      RowBox[{"onesol", "[", "ip", "]"}], " ", "-", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"model", " ", "/.", " ", "fit"}], ")"}], " ", "/.", " ", 
        RowBox[{"t", "\[Rule]", "ip"}]}], ")"}]}]}], "}"}], "]"}], 
  "]"}]}], "Input",
 CellLabel->"In[126]:=",
 CellID->419081260],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWktPFEEQnmVmF3ywgBxEvKwnEMV4QDixgERJiJ4ML2NMCDGBRFmD+A/U
GB9HE13lYOIrHEyMCR5MjAkXiQev6tE/4A+QpJ3unt6arq3pnt0dcA1s2Nnp
6a8eXd31TU8N52eX569cm11emJvNjS3NXp9fmLuRO1tY8i+5KcdJzTuO87XD
8c+Zfya+/LPgf+Vp9FnGP7q+gMtbaSXJRF9w1Qtf5dgG/7chuDgukHv40S3h
nHSozfWkdT2BjrAEP4zJLjcQ4ij/6yFkBrU9q+4sYatfQBuDkTRV7C/XqQCn
BKSJETEoi6jd20zgEx4neI2j3WzVWT6eNKH9uLg2nC8ODT4ZGiyK6MAKaQn8
ckO/XilYIdWk7mPCg5cvvgnd/E9q7zaMtke4fnN5LZDJF0fyjwOvumKN6KjQ
ELb6Ye27nKkIDUpBN5LMF+/d/Swjb/RZyHlTE698oeGhom9ur5A5GYV1N778
Kiy+537ti0T2x+hRHXGwyfbUYvmIuCbj/PDBunPAgpSRPTP6zIqEuEYjo8bA
Z9ZTeVQhK7YRdjrRKA8xxT5h1sHteCxE24I4SVuYVV0k2Yja1bFqJ4q8tF37
3SGawbFuj5yxylhX5/VMmWyLVVs8vu1l+qoQlkr3pNr4VuqGVSB1m5irHXlz
EM2l1BCPd9uR9ZxBVom2I3s5q8c0q4yicfw2oMBDEwp8ApSd8/YjP7osSPDF
hgR/aOQMsvzDgAKrgOKgGWSJ1mGOQHRPGnl4woIELwGpgGnkKa0L59pH1JZ7
z0YGXF4r7+MMXEUWs6hfeoB3keF1Xx1X9qL4rCK7WdRP78KTv1/8RHanExw/
B/xB8abZdqt37Zso2jQvYy+yVnv2O8wmii9eB0nebWSmR3F1AUV+XaDtd5IC
ipVJjosV0AjXjV6B3/G5awONZMGAAr8BxUEbyEtaRzXeRfdMh2Y1nE382qSI
pz8mR33wWuOoKRZiRqShTRvfVNn8YK64EIoPndNTSALPXZYcZVzLMvbba3kC
td0yiaQsp4O5nq7YosdCd7KYlsoQFVsMt1tJy2r1Kq6aZJDRam2G+0ELV/Ka
6Tl7n7QhUZCzJhTkLo3ih+gMFfsT99LFN76lc+PPA4LtCSI0JDIQy7WiNt5r
VNq/1fha9YmaiXvn9qfTI08Xr74TKynld6fIqO606LQao3EZtfuYvv67LHi6
WkJrhXxZKUlx0COm58kKqZMf/p9q1S0Ux7cGFMTFhIL4AKpa77YvLtozksbu
+BmhgdkrY1InuQsP7TLKazDxK0SmJ4qSrYp0H2b6SuAXOxIcBwfgKl60Dbri
ZR9V1BOLHB2szWjLSTylwDxtx5MIIOLVsUxYPV+SfMqI21P/XPHvfNIq+Amx
FK7P49pEfbEU6CZr5An5jXXT75ErYyMl3leaya2pv+/yjt6DK9b8Il1hxhVr
G1K/mwByJzJYXMvaf2YkxGDbyQR4XdfCYCpHSQZOiHGT2Efpbw636v809PWx
y1txe+o/5+vRJ5EnZXlBvyHzWD088alMxDXYWhgI7/WqYyJ9Z6NYLUPGsvon
OPBaq4gmxjx47ppZcgxUejPsn6s9yACjWUW+lfN3IY76gL4INN+J+OdqJzKg
2U6Wi5zUX82t2yk=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 203},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[130]=",
 CellID->639613595]
}, Open  ]],

Cell[TextData[{
 "Typically the default values ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " uses work fine, but you can control the chasing method by giving ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " the option ",
 StyleBox["Method->{\"Chasing\", ", "MR"],
 StyleBox["chasing options", "MR",
  FontSlant->"Italic"],
 StyleBox["}", "MR"],
 ".",
 " ",
 "The possible ",
 StyleBox["chasing options", "MR",
  FontSlant->"Italic"],
 " are shown in the following table."
}], "Text",
 CellID->167600188],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
the numerical method to use for computing the initial value problems \
generated by the chasing algorithm\
\>", "TableText"]},
   {"ExtraPrecision", "0", Cell["\<\
number of digits of extra precision to use for solving the auxiliary initial \
value problems\
\>", "TableText"]},
   {"ChasingType", "\"\<LinearChasing\>\"", Cell[TextData[{
     "the type of chasing to use, which can be either ",
     Cell[BoxData["\"\<LinearChasing\>\""], "InlineFormula"],
     " or ",
     Cell[BoxData["\"\<NonlinearChasing\>\""], "InlineFormula"]
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->2116124712],

Cell[TextData[{
 "Options for the ",
 Cell[BoxData["\"\<Chasing\>\""], "InlineFormula"],
 " method of ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ". "
}], "Caption",
 CellID->1595931707],

Cell[TextData[{
 "The method ",
 StyleBox["ChasingType->NonlinearChasing", "MR"],
 " itself has two options."
}], "Text",
 CellID->1120970323],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"ContourType", "Ellipse", Cell[TextData[{
     "the shape of contour to use when integration in the complex plane is \
required, which can either be ",
     Cell[BoxData["\"\<Ellipse\>\""], "InlineFormula"],
     " or ",
     Cell[BoxData["\"\<Rectangle\>\""], "InlineFormula"]
    }], "TableText"]},
   {"ContourRatio", 
    RowBox[{"1", "/", "10"}], Cell["\<\
the ratio of the width to the length of the contour; typically a smaller \
number gives more accurate results, but yields more numerical difficulty in \
solving the equations\
\>", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->365746491],

Cell[TextData[{
 "Options for the ",
 StyleBox["NonlinearChasing", "MR"],
 " option of the ",
 StyleBox["Chasing", "MR"],
 " method."
}], "Caption",
 CellID->1115642365],

Cell[TextData[{
 "These options, especially ",
 StyleBox["ExtraPrecision", "MR"],
 " can be useful in cases where there is a strong sensitivity to computed \
initial conditions."
}], "Text",
 CellID->327764883],

Cell[TextData[{
 "Here is a boundary value problem with a simple solution computed \
symbolically using ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->1710768027],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"bvp", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
       RowBox[{"1000", " ", 
        RowBox[{"x", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", "0"}], ",", 
     " ", 
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ", 
     RowBox[{
      RowBox[{"x", "[", "1", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{"dsol", " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{"x", " ", "/.", " ", 
    RowBox[{"DSolve", "[", 
     RowBox[{"bvp", ",", "x", ",", "t"}], "]"}]}], "]"}]}]}], "Input",
 CellLabel->"In[131]:=",
 CellID->1523215861],

Cell[BoxData[
 RowBox[{"Function", "[", 
  RowBox[{
   RowBox[{"{", "t", "}"}], ",", 
   RowBox[{
    RowBox[{"Csc", "[", 
     RowBox[{"10", " ", 
      SqrtBox["10"]}], "]"}], " ", 
    RowBox[{"Sin", "[", 
     RowBox[{"10", " ", 
      SqrtBox["10"], " ", "t"}], "]"}]}]}], "]"}]], "Output",
 ImageSize->{285, 22},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[132]=",
 CellID->70995793]
}, Open  ]],

Cell[TextData[{
 "This shows the error in the solution computed using the chasing method in \
",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->62388872],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sol", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"x", " ", "/.", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
           RowBox[{"1000", " ", 
            RowBox[{"x", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", "0"}], 
         ",", " ", 
         RowBox[{
          RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ",
          
         RowBox[{
          RowBox[{"x", "[", "1", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], 
       ",", "x", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", " ", "0", ",", " ", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", "\"\<Chasing\>\""}]}], "]"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"sol", "[", "t", "]"}], " ", "-", " ", 
    RowBox[{"dsol", "[", "t", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[133]:=",
 CellID->1787592186],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 200},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[134]=",
 CellID->301569196]
}, Open  ]],

Cell["\<\
Using extra precision to solve for the initial conditions reduces the error \
substantially.\
\>", "MathCaption",
 CellID->767104824],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sol", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"x", " ", "/.", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
           RowBox[{"1000", " ", 
            RowBox[{"x", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", "0"}], 
         ",", " ", 
         RowBox[{
          RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ",
          
         RowBox[{
          RowBox[{"x", "[", "1", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], 
       ",", "x", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", " ", "0", ",", " ", "1"}], "}"}], ",", " ", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<Chasing\>\"", ",", " ", 
          RowBox[{"\"\<ExtraPrecision\>\"", "\[Rule]", "10"}]}], "}"}]}]}], 
      "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"sol", "[", "t", "]"}], " ", "-", " ", 
    RowBox[{"dsol", "[", "t", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[135]:=",
 CellID->1132997041],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 200},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[136]=",
 CellID->581476063]
}, Open  ]],

Cell[TextData[{
 "Increasing the extra precision beyond this really will not help because a \
significant part of the error results from computing the solution once the \
initial conditions are found. To reduce this, you need to give more stringent \
",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " options to ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1500453481],

Cell[TextData[{
 "This uses extra precision to compute the initial conditions along with more \
stringent settings for the ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " options."
}], "MathCaption",
 CellID->799445058],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sol", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"x", " ", "/.", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"x", "''"}], "[", "t", "]"}], " ", "+", " ", 
           RowBox[{"1000", " ", 
            RowBox[{"x", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", "0"}], 
         ",", " ", 
         RowBox[{
          RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "0"}], ",", " ",
          
         RowBox[{
          RowBox[{"x", "[", "1", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], 
       ",", "x", ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", " ", "0", ",", " ", "1"}], "}"}], ",", " ", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<Chasing\>\"", ",", " ", 
          RowBox[{"\"\<ExtraPrecision\>\"", "\[Rule]", "10"}]}], "}"}]}], ",",
        " ", 
       RowBox[{"AccuracyGoal", "\[Rule]", "10"}], ",", " ", 
       RowBox[{"PrecisionGoal", "\[Rule]", "10"}]}], "]"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"sol", "[", "t", "]"}], " ", "-", " ", 
    RowBox[{"dsol", "[", "t", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[137]:=",
 CellID->1416681447],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy1nVuIpdlVx89UnXOq63aquqrr1tPVdem6dVV1T18maRV0HkQliMRbiCA+
zAzCDJhMnExe9EVfzIu+zIgaEPPgNRdlRIgIEYKBYDQ6IgZEBYkgEc0oGHwY
EI5rf2vv/f/O7+yzq2vaFD1V5zvf/tb6r7X+a+3Lt79vfuD51176yQ89/9rL
Lz6/+32vPv+Rl15+8aO73/vKq/bV9FOdzlMvdTqdv9js2OehfWr+uzccDjvT
4VPz0/x5b/jQ+dae6Ydf3c98+m+f+85P/OzP/Mk77/xv+Pq82LbX/Pryn//L
x177nH20i37pF7/YuaB9/8UXPvuP//AN/+onfvxTX/vaf31bo/PrX//vVz78
R6bz7bf/5z+L13eb46yvM9Ucm9o3Xv9S+NhY1DfIJudPP/9Pf/y5vzf5dvh2
RdrP/9znramkffOb7/zQ+3/TsLi0bpZuP1Of/I2vSFYQNd1cYyq+/32ftN9R
5sd/4QtmZPj4bDr/wQ/8jpkVddgnOw7tv1FENg07u0nK93z3r0vqW3/9r2an
2pvv//DNr4aP/1GRasaYSTka02aswQkfHybs5rcSippUA2JwJDXHOEqVFrfF
3GWODh//vSJVV/USNqNH+PigOf7q3/3byy+9qfOKZVlqjrCpz1wPx9GvD2KM
/fsv/tk/R22Pkr+jnDcr0uWvGVh6v+gpRfU9zZd2bNdbK3OhXVjV9du/9dYn
fu3L0iWv30uyzYVBygz82TCol2NtoE1jVZe5whwiXQbSrpIuUxw5P4NYPMT5
KVz/BxWtliaWipJq7LX4SKsR3miv84rZQ6B2eSbMRI5qDUobwjkxOrNgxT3k
xQwkPUjtzb2qAUL++7Kvn+2zH8/fObDkGUTqSpFFBsUADXM+2XGsBZO0+RVz
wHYXGTGL2NzHca5EVlhDjZykzWhp5JwD4+8iQ1ybtx1hbaymfdj62QnaPMpz
iJNrU+8wC2muTVW0j7hO0uZ91hyw34V216YKFeSE2nIFvPyMtPSyFi8KnfmE
yOggHcryzNQYvXvwtzNVVaKoyxOzswDe3QHv5nD+LmLldinLi7qsIlhdiLrc
j+FjM1oIvWK0cx6suwvZs2Dxp0u6PDRRl3xyDg66LvWKrkt1xXWpmkmX1w1F
ch6RPQf/5iHZNanXnkX7olXutGiV2HcOK+bBvjJz5NGiLqNgiOwCcJ0jOvNg
4R0wYw7HLV1d1+U5twgWnINxZOg5bJwHqz41psejPgB7ToHfcZhQEz05jsr0
cT0e4wHkuB4DZxClR3LOwYd56B3X4/EegHeuRz2h61EVOAcucmVcj8seAO8p
5C5Cr+tRBZlPPIj96rgej90A3D6F3kVw9wy8WQQPfk964vjCpjadJaA/hVcW
gda1qG4swuvSEpQswRen8MUAXD5FrBfhA9jgMKIWjTxug8lLyNRTMMy1eFEu
aHE4nWXkywn4sgSpZU+qPbQ4iaMW1ekTYFuCh+hRxgta3AlRi7gUtIReeRna
yp5UvH53VHqsyiXp7b5jCRXgdtGTipdr6Q9bc81eeySSBq4pf/wntA2j3Sn7
ezUhNMqHj8vR3tuIkSP3RkKuvooxdmSzrZxqwLWOA7beKLaIq33FoOVB4VCU
TxBlP6/c9vPq98gSx+lO8p/glS5Q9HHcfXzcMb8dl/L9GPy5WmS3Is168f/p
326uR2Ke41MNdXxkrvzM6x3fTGTZTJGZF3ux174+fB/Z5WiUbUfwjp9X7Xa0
Yq+jVS0Q2rb2pQvx8YrMmxg1x6EsOgL7/LyqintdMww/r8rajJy6Jsza2z/7
9mMNsCvRT4P4d7r1txtxTyc+hA8xalfBMsenbPHzqj+OT30cWdv06H2fa5pV
9s/oYM1fa1Aej/kz4/FVxDGWHQGfn1dP7nhUhXj9WcJveEL/N5XiHlfwPnoR
rvhNkBuq4jFYv9Ic+zB4sv/EwzNcP93yb8D36rvCI/0rkO94NAa5ijzhqKKX
2sc8++lG/UlkE6ugS9N8+RDRWS2yS3MMv1415hTnHY3XyOAdR8N1INp0CM46
CvVMR/DZ1ZaW8PE2OO45rpHWRyooVP8chTJ7FUwo+0qoThC59ppSWK8Tiunc
i2sGdQhtq0Dj57WysQre5N4pjhtn0/no41eKPnD2qdodwPOr0OoolL3kyzHY
meeksTrWUGhMdQCpqy1PCoXi6+dVQxyFalxej4vnP1xBISkH8PC15lg90gHi
HVAow3U/YD5FNubMZO3h6kOwyrWqP3Gtyji3XRl9hGPO5Gu2K663cNUaPHoA
lI5CGcgqlleHYsQ+VEEhKbcSe6Ln1hChW8nTMSPdV8roI/huIcmP7WsopNW1
qFqtxUi5D1S1rsFnhzgegD8/VdGuq24hMmvFSCnea4nrcXR9gPaDxJ+Y4WUU
rG370eb1YoSUiWspzjFnDsCbASIg7UH5KuK1j3ivFSMjlvl51Y3M4lgTl2GV
r8tbqYpj0m+v+EI27ON4rRgx1dQyaxWR5RSRKO+ZhDJaVUMlqfvw3Xohgm1e
jLJY1uRZc6y1eU3YiBvm9N9RQSMe7IPr6+DFHny2Dp/49cpsjtp8PdJMNRXh
LlcNlXqjvXRVtG0Dtu+BX2T7PuRx7HYn+c4oGdb6a6hUT1yr6sgGIruH9utA
ST46KnNkHEn4DMRj41hrjBKOvWRNHIFsRMbQC+uI5V5iTBzZrKbjOHNrZhw9
i4LfDfVuv8py8WAPujfAGz+v0fwGeLPbHGs8w971DL4M9paRcTSwiwzYAHcc
mWrTBrjD610+VoD7mQX245+FLoCjPS5VI6sNcGcPKJgRuzi+Bn87qoxymNb9
az5TZaT0jSKTFD1mxg6Y5pWWa3F+HM773E018FEFpXy2C59uAtUOOLqJSO6A
c+tA0R4vWmTiioSYUkMppu8C1SZQOwplt59XJoXzge0b0H4MHnZhXQ2dZjP0
0RYi7+dVazZxvIP265B/BF73cH0NJbVIylaRb9K6CZ/uILKs1BwVO0r1RzWU
4o9rUX+wVeSb5nCb0HITqDzinImqfR/nayh1laNQpdjC8Q545udVT7fR3q3g
TFX899m2Kll5J9watNwY+jpJYH5aUW5Ynkej8nNq26RAvDuQdouxx9I659P5
KvFIa/mrsGJ7qFXARqoPXjtaUVVdCtUt3RcY2WWTF8fL1o9ia63x2gcDOLKS
3skYxZfmqn77Kq86XIH36zRTvT6mLXgQ9zrsg49flJU3xlr4XZMwGm19GXlq
jDFShfWWU+AZ1zSVvRSarQHvuHcsB8fW9RytKtv4VSF+aVXXs0Ssv97CFDCX
VthdgzJrkj8sN0b8kVfBYt24fQl/jLNGVWvcwhBHrjRyxD4e/YC3nT2dHAPV
rrZ/TH3RP6WMK90DUAUNGdLsbemazw2GOdaEnzyRf4R53D8hsmX/qH8e90/Y
VROsnYp+CTWqbVsoWe4PjfwQpYIXvIJ2CnWjQdgz8hs9rULa78D3oyfyivrK
Np7Qi5Qi6Taolx/3iQkb8Ul75aPNlTAnHF+D53ywwTTblj9M+5kiH+mjhjU9
c7GVCesOzd1hxHbwRD6SBbmHmfYBbJIla9WH5H6l1XuUK7DmILombgGP16gn
Uo7knrFp1N7fHb5tVtLCrmzjvHHYXGGRMZrvDXXf4aJ+SD3rNnQ3cWmNaPy8
uN1Emj1Saxzmx5px7kBfk3u9jN5vePk8eQJWyd5GzFyXxil+XuMQx4r+ocXB
ppq3eh0fnamHyHkUd2eWd+GvQUszjrACcQNoHZ3GP44GVbrFMr9+dHwyPnb0
9f1QLc6K6LgW5VJRMVs+9vOaB/oxamXLZ2UrxZc8Ao47Rk4rKBW5G4jE00lK
ux8eQ63qtQUrHJVmpzdwfD21jyibXDKPNXW5wRwqyXMhf8ewL+B4BnFkHVss
ttfMhe0p3+tgjohDi/foKVV9wONJzWyLUgdo1dQW4zare1lae22iJM0x6m4N
x/OU2tzRCmP5mBGxFs6aI8LW1E7HtFlMTZS50vipnfohpp1O6LU9oo8Tw2An
ETF2eX4Wq91FFrhf8kyqGjvl6OPFLse6Ert2jrDHKkkNPrhIO+/4XuSD5o5g
15gWBhTNXga/T2YkCBsJQ79nf6eaobs/NNWY11xlbrOMtSYW3XIkvZKwj1Dl
IDqvoprZeXtVHlrv7blaohk6M4NrkVupfXyCiVHwkZS53fD7uLph/bzx2jhm
v81yo7llgDUwJ9pn0xyY7WM6O7YQB8lxFjLBQ6Mjk/YIYBvx9POszWrvHlC/
tNMcq/a6xep1b4HdOUPNIyGNm+MrpsyuNxX2z3RbW1+ynq3YpHr+NKK0DQx+
fnQtos0SrtDtwgebOOZ9SmZwk6F9s9HcZv/MWXatiZ+rWCMPOlrV8m1Ydx3o
txPHYlVytOqT3Rr1UBtAfwj0XjnU5y8k+RFvOSZc/3eUHM+IR9dx7KMx9Ysb
OHYrNPrj+vlhQhlnU/n5sPgM11YFNceQ8t02fO3nVW9u4nquQ5bvc1ljOw71
I+8lMUO0Ii1m11Bj9tZasbsJvgcpobbvNN+r2q2DCfuIIWdCzbw9PGlq6Ey7
weZTe8Ib4PIuguPU6NJxaj3Ba4eq801cvx7tuAX7824Ec0GIdTM27rd7SM/t
6xVvqgJeh/duIh+2Ct7Wer7apTv51xITzbDQCfo+QPeiNzVLzIU1dKpNW2Ak
Y7oFn95E3pTvoXO3k7U3B4Tli2au1nPf+m6C0NvUsHK9nKv2XC+fPHtcA7tu
pfYxp1dS3O3YRMb7+abPfGP/vJJOQClUjnL03kSI21aSHnPR0Y3ukdE+ItWM
5r5+13PaCXkvMSraUfOeIsue1PWr9mzCW35eM1Pu0TiAvOXEewMWak9+wra9
YjqGchPZ4ceq8zvgU7lv1fWeHdxFIy8vtfgY1szfX0El1vuxKskOYknUjkqV
aRXtj4CymRl0zUpTaeEwStZwqVbweAcZUO67VenKe9AU46av7vodf0Nr3vjB
jCvAonzeu9yFnaxpu/Ar76ocxZxYaOWABS48hFnxDysC7/VLf3lUIP28E3IE
Ns4n/KYi7IX44YmoghWOhnsi2H9O3jPBHWbHKdojz/b5aCA8K/QjFR8p5qxf
e/DhevG86pujUt04ASdn4bMaKrVi7d8btntFaecenBUwks+kXAaNIsFecw8+
5F1t7lfiPZUTWPd4qLYgxbVqpLEHKeuIJPd2LQMln7O6DCruX1NO7yVWxtq3
DtZz/5ujwk6VVg5euQQq7jvgjjFFeB0ouDdrGbw6Ba8ug0q+KWtVv8O73vu4
nvsi+bTpZFQB1BZsWC9q47hYfOEeUj6XdAY+XcZHyiWukN+CD7jLgrs0l4Dy
HHzKKwiPgUp1h3cXbsGXjkoVgnP8JUQ+oNIbCbRKWUMj7bwv49pUr67BB9yf
PcDxHfg036GP1v9oBRVHjFy7EUru1+Le9UH0Sd7JGsfpXIsso+G8i/uu6APu
ceNOeT6JzndM9NC+hkpx593f8s54VQQ+27CI83fBU18BUM5+oIKKmT7ZFysp
InGceYjr8379yM9nUnvzddhz0AUTaqjk4bIv5Ms8fxt5LkjX5+c5rPMJexLu
p+NYaabB18moRubhE3zAcdHhsDWWDVYYNcPc9z4i7ijUawtFAMEVI84sDpFT
K7CJzwHNpSzw6WR8U4986isyqpa1SHHWz6ePVH9WkIuHiS9xRjuXfGDtw8j2
IaxyVMrtGipxn7OdQxxz7HiYfBMrBN8O1KDqZ9TDtIf2g0U0XDVOzw0dIYIr
8DifXuPbQvxNVTkr3JxODQMzRhnCJ19WYDGf57sCDzqWzLb4NoEaFuaLjqlr
GX44hi1XwCnHYsD9KT4XHLGMPMU/viOrhTI9zTv2DP+IFcpW7g5yK1Svy8+m
y2q3QqNjf++dc9Lj4BbwaXTu/euPWTbxqfkRS1QNV8DXIyBdRvtjcOkK4tdY
EmqANQm34X6saAfvgF3irQUjdqgH9IipChDnMiIYzmskplx4lOSaSSHDyvjf
XRy6ubJrTOe4VUOPgdNxqyby+ewZyMvvzYsrX44/vTXA95BlVBH9xd4fe2vA
SBSUrXx++RjscWs04uDT7zM4dmssqcyaMKKQNW0UT/BWgXjs9/ZG58yBHSew
x/FrbMJ3OswgOo7fUsNf6xnxX/rdAiMoOYce5XTbe0s4Pklo4lgm7WB4o0F1
8TPy3HHj+j+O53qVg3zfz214rw/8r18Kh/oAx8E3ffDNP8LpONT3cqZxORx8
UpBv7ZG/+Nao2zjPe6iOY9K7Ari/iasErl1jwgGk34ZX+rjetbM/d63KeHLM
parnGyAWt4ft931KzhtFbdvFSMqm9hNE4WN+u9LI2xiUvz34TFqDUtem6jQA
P7lytggeujbVcj7lUrNRcRiAFXzr0kLxvLLOtSrralrJDuWAS9WIZgG+8VUf
VcwufFXTKraU3wvH59L5/B/fbKvesaaVbyvTivrZsD1n8/1D4eM5fNKFD8ra
uCdhASw4wzHfhniO+HVhc02rsPFtha5VPRXf93iOTJkG13+5olVRX0D8zsDN
eaB0rXyrsvKzplXcLb9vUtzkG1jLWlWzalrF3fIbNRX32SR1ZLWNu8zUe0hr
UHoTnF2AdK5v8p2ZZW06X7ZxB1rmEI87iO8sPH83eX5kn6riXdMqT84hI+7g
eBYZ4Vqt54p8mIIVNa3KgDkc3wHL8rp2HD01q13hzUweX/vxaP1KRZvycQ6e
uQNPc4W4kaf5f3wbYU0XV3/wzquxFXv1QK4rZ3UYIb7+paouvidWtZXvVubO
Z9fl1zuisPZb06Xxa357a7TzGbCTuu6n8wYn7CL81aIW7uPj24PzWylGnmyk
FnN15FFNi/LRtSifXYt6CWr5yyw1CN0tcpVvuOY8pSRtHKPqJOea9xCPy0gV
+2aA/T480X8sqXvFqxS1+4jaZaSqf+P+wAeI0mWk6qoeovMgMWjkCd/Hk6oe
owe/PgQn/LwqUFnqPvzahV/5zvcuovuVilTuB5df/f+2oArWhcckNQjdBy9d
msYyz8Iz+f8YEf1Zxuj3aLhGrv7hPTiegg1/NVnqyC5Dn708AuYpWFyTZtpM
ghdYj8QjWMx+uCytudMS9idZuC3l07s1ejn6TsnOW8Wr83PpcSzg1+bqEd8N
7tfODFvPRj/RyuwLjQQTbmh9u7xruIJ2fI/n5VdOX0jxNWaHlcG/KerhLP3y
K5svDiM7Y0Z9q+wJv95blN3kx7tYJUxPFYa/6fJnixoGF8oa18bVF+FPq5tL
w/J6ml/aElWRlVZbwjfPNd+MrwOFP9914blxSeHTy97sqf8DIER/kw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 202},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[138]=",
 CellID->188794340]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Boundary Value Problems with Parameters", "Section",
 CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
 CellID->3518691],

Cell["\<\
In many of the applications where boundary value problems arise, there may be \
undetermined parameters, such as eigenvalues, in the problem itself that may \
be a part of the desired solution. By introducing the parameters as dependent \
variables, the problem can often be written as a boundary value problem in \
standard form.\
\>", "Text",
 CellID->502458878],

Cell["For example, the flow in a channel can be modeled by ", "Text",
 CellID->153136180],

Cell[TextData[{
 "          ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"f", "'''"}], " ", "-", " ", 
     RowBox[{"R", "(", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"f", "'"}], ")"}], "2"], " ", "-", " ", 
       RowBox[{"f", " ", 
        RowBox[{"f", "''"}]}]}], ")"}], "+", " ", 
     RowBox[{"R", " ", "a"}]}], " ", "=", " ", "0"}], TraditionalForm]], 
  "InlineMath"],
 "\n         ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"f", "(", "0", ")"}], " ", "=", " ", 
     RowBox[{
      RowBox[{
       RowBox[{"f", "'"}], 
       RowBox[{"(", "0", ")"}]}], "=", " ", "0"}]}], ",", " ", 
    RowBox[{
     RowBox[{"f", "(", "1", ")"}], " ", "=", " ", "1"}], ",", " ", 
    RowBox[{
     RowBox[{
      RowBox[{"f", "'"}], 
      RowBox[{"(", "1", ")"}]}], " ", "=", " ", "0"}]}], TraditionalForm]], 
  "InlineMath"]
}], "Text",
 CellID->125679040],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]], "InlineMath"],
 " (the Reynolds number) is given, but ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " is to be determined."
}], "Text",
 CellID->66336125],

Cell[TextData[{
 "To find the solution ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", " "}], TraditionalForm]], "InlineMath"],
 " and the value of ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ", just add the equation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"a", "'"}], " ", "=", " ", "0"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->282084070],

Cell[TextData[{
 "This solves the flow problem with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", " ", "=", " ", "1"}], TraditionalForm]], "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 ", plots the solution ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " and returns the value of ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->480930858],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"R", " ", "=", " ", "1"}], "}"}], ",", 
   RowBox[{
    RowBox[{"sol", " ", "=", " ", 
     RowBox[{"NDSolve", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{
          RowBox[{
           RowBox[{
            RowBox[{"f", "'''"}], "[", "t", "]"}], " ", "-", " ", 
           RowBox[{"R", 
            RowBox[{"(", 
             RowBox[{
              RowBox[{
               RowBox[{"(", 
                RowBox[{
                 RowBox[{"f", "'"}], "[", "t", "]"}], ")"}], "^", "2"}], " ", 
              "-", " ", 
              RowBox[{
               RowBox[{"f", "[", "t", "]"}], " ", 
               RowBox[{
                RowBox[{"f", "''"}], "[", "t", "]"}]}]}], ")"}]}], " ", "+", 
           " ", 
           RowBox[{"R", " ", 
            RowBox[{"a", "[", "t", "]"}]}]}], " ", "\[Equal]", " ", "0"}], 
         ",", " ", 
         RowBox[{
          RowBox[{
           RowBox[{"a", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", "0"}], 
         ",", " ", 
         RowBox[{
          RowBox[{"f", "[", "0", "]"}], " ", "\[Equal]", " ", 
          RowBox[{
           RowBox[{"f", "'"}], "[", "0", "]"}], "\[Equal]", " ", 
          RowBox[{
           RowBox[{"f", "'"}], "[", "1", "]"}], " ", "\[Equal]", " ", "0"}], 
         ",", " ", 
         RowBox[{
          RowBox[{"f", "[", "1", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"f", ",", "a"}], "}"}], ",", "t"}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"Column", "[", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Plot", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"f", "[", "t", "]"}], " ", "/.", " ", 
          RowBox[{"First", "[", "sol", "]"}]}], ",", 
         RowBox[{"{", 
          RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}], ",", 
       "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"a", "[", "0", "]"}], " ", "/.", " ", 
        RowBox[{"First", "[", "sol", "]"}]}]}], "}"}], "]"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->753178145],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->609963450]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->43850463],

Cell[TextData[{
 ButtonBox["Advanced Numerical Differential Equation Solving in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
  FontSlant->"Italic"]
}], "RelatedTutorials",
 CellID->604721514]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Numerical Solution of Boundary Value Problems - Wolfram \
Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 40.8055016}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "The shooting method works by considering the boundary conditions as a \
multivariate function of initial conditions at some point, reducing the \
boundary value problem to finding the initial conditions that give a root. \
The advantage of the shooting method is that it takes advantage of the speed \
and adaptivity of methods for initial value problems. The disadvantage of the \
method is that it is not as robust as finite difference or collocation \
methods: some initial value problems with growing modes are inherently \
unstable even though the BVP itself may be quite well posed and stable. \
Consider the BVP system X'(t) = F(t, X(t)); G(X(t_1), X(t_2), ..., X(t_n)) = \
0, t_1< t_2< ... < t_n The shooting method looks for initial conditions \
X(t_0) = c so that G = 0. Since we are varying the initial conditions, it \
makes sense to think of X = X_c as a function of them, so shooting can be \
thought of as finding c such that with ", "synonyms" -> {}, "title" -> 
    "Numerical Solution of Boundary Value Problems", "type" -> "Tutorial", 
    "uri" -> "tutorial/NDSolveBVP"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:3"->{
  Cell[1436, 49, 97, 2, 70, "Section",
   CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
   CellID->659822336],
  Cell[122151, 3474, 119, 2, 70, "Section",
   CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
   CellID->3518691]},
 "b:9.0"->{
  Cell[1436, 49, 97, 2, 70, "Section",
   CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
   CellID->659822336],
  Cell[122151, 3474, 119, 2, 70, "Section",
   CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
   CellID->3518691]},
 "ndsg:3.1.0"->{
  Cell[1436, 49, 97, 2, 70, "Section",
   CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
   CellID->659822336],
  Cell[122151, 3474, 119, 2, 70, "Section",
   CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
   CellID->3518691]},
 "s:5"->{
  Cell[5547, 181, 93, 2, 70, "Subsection",
   CellTags->"s:5",
   CellID->468358538],
  Cell[13304, 479, 85, 2, 70, "Subsection",
   CellTags->"s:5",
   CellID->224929489],
  Cell[44026, 1286, 74, 2, 70, "Subsection",
   CellTags->"s:5",
   CellID->519011742]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:3", 132187, 3758},
 {"b:9.0", 132417, 3765},
 {"ndsg:3.1.0", 132652, 3772},
 {"s:5", 132880, 3779}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 703, 17, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1328, 44, 83, 1, 70, "Title",
 CellID->1974382719],
Cell[CellGroupData[{
Cell[1436, 49, 97, 2, 70, "Section",
 CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
 CellID->659822336],
Cell[1536, 53, 642, 10, 70, "Text",
 CellID->96799541],
Cell[2181, 65, 922, 30, 70, "Text",
 CellID->382453414],
Cell[3106, 97, 734, 24, 70, "Text",
 CellID->177552007],
Cell[3843, 123, 994, 36, 70, "Text",
 CellID->254723824],
Cell[4840, 161, 682, 16, 70, "Text",
 CellID->584806262],
Cell[CellGroupData[{
Cell[5547, 181, 93, 2, 70, "Subsection",
 CellTags->"s:5",
 CellID->468358538],
Cell[5643, 185, 72, 1, 70, "Text",
 CellID->622958117],
Cell[5718, 188, 1661, 59, 70, "Text",
 CellID->410269708],
Cell[7382, 249, 757, 27, 70, "Text",
 CellID->115473309],
Cell[8142, 278, 568, 22, 70, "Text",
 CellID->206138752],
Cell[8713, 302, 1179, 44, 70, "Text",
 CellID->26324775],
Cell[9895, 348, 1086, 39, 70, "Text",
 CellID->306888864],
Cell[10984, 389, 575, 21, 70, "Text",
 CellID->544456068],
Cell[11562, 412, 181, 7, 70, "Text",
 CellID->180742921],
Cell[11746, 421, 797, 25, 70, "Text",
 CellID->45412943],
Cell[12546, 448, 721, 26, 70, "Text",
 CellID->241406211]
}, Open  ]],
Cell[CellGroupData[{
Cell[13304, 479, 85, 2, 70, "Subsection",
 CellTags->"s:5",
 CellID->224929489],
Cell[13392, 483, 459, 8, 70, "Text",
 CellID->19996186],
Cell[13854, 493, 274, 7, 70, "Text",
 CellID->159580041],
Cell[14131, 502, 170, 4, 70, "Text",
 CellID->113392259],
Cell[14304, 508, 494, 16, 70, "MathCaption",
 CellID->603352701],
Cell[CellGroupData[{
Cell[14823, 528, 1847, 52, 101, "Input",
 CellID->49231136],
Cell[16673, 582, 4390, 76, 127, "Output",
 Evaluatable->False,
 CellID->30064504]
}, Open  ]],
Cell[21078, 661, 258, 5, 70, "Text",
 CellID->163787919],
Cell[21339, 668, 79, 1, 70, "Text",
 CellID->473868030],
Cell[21421, 671, 2132, 66, 70, "Text",
 CellID->563834588],
Cell[23556, 739, 55, 1, 70, "Text",
 CellID->16284262],
Cell[23614, 742, 854, 24, 70, "Text",
 CellID->148768874],
Cell[24471, 768, 1332, 35, 70, "Text",
 CellID->44686511],
Cell[25806, 805, 187, 6, 70, "MathCaption",
 CellID->795575749],
Cell[25996, 813, 2884, 86, 70, "Input",
 CellID->232407840],
Cell[28883, 901, 229, 7, 70, "MathCaption",
 CellID->7358722],
Cell[CellGroupData[{
Cell[29137, 912, 815, 26, 70, "Input",
 CellID->183247138],
Cell[29955, 940, 410, 8, 70, "Message",
 CellID->32853519],
Cell[30368, 950, 381, 8, 55, "Message",
 CellID->733380702],
Cell[30752, 960, 2752, 49, 122, "Output",
 Evaluatable->False,
 CellID->610290743]
}, Open  ]],
Cell[33519, 1012, 1268, 38, 70, "Text",
 CellID->120337663],
Cell[34790, 1052, 202, 7, 70, "MathCaption",
 CellID->4681285],
Cell[CellGroupData[{
Cell[35017, 1063, 1902, 52, 138, "Input",
 CellID->65999615],
Cell[36922, 1117, 2335, 42, 122, "Output",
 Evaluatable->False,
 CellID->5245260]
}, Open  ]],
Cell[39272, 1162, 384, 12, 70, "Text",
 CellID->97834841],
Cell[39659, 1176, 241, 7, 70, "MathCaption",
 CellID->47675345],
Cell[CellGroupData[{
Cell[39925, 1187, 1634, 47, 101, "Input",
 CellID->54666566],
Cell[41562, 1236, 2415, 44, 122, "Output",
 Evaluatable->False,
 CellID->171794514]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[44026, 1286, 74, 2, 70, "Subsection",
 CellTags->"s:5",
 CellID->519011742],
Cell[44103, 1290, 1497, 40, 70, "DefinitionBox3Col",
 CellID->508999769],
Cell[45603, 1332, 61, 1, 70, "Caption",
 CellID->18856673]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[45713, 1339, 53, 1, 70, "Section",
 CellID->948663323],
Cell[45769, 1342, 759, 18, 70, "Text",
 CellID->262954798],
Cell[46531, 1362, 59, 1, 70, "Text",
 CellID->847545622],
Cell[46593, 1365, 406, 14, 70, "NumberedEquation",
 CellID->119919937],
Cell[47002, 1381, 1045, 42, 70, "Text",
 CellID->400908072],
Cell[48050, 1425, 393, 12, 70, "NumberedEquation",
 CellID->947916108],
Cell[48446, 1439, 713, 27, 70, "Text",
 CellID->887862838],
Cell[49162, 1468, 642, 22, 70, "NumberedEquation",
 CellID->21985348],
Cell[49807, 1492, 42, 1, 70, "Text",
 CellID->1301566606],
Cell[49852, 1495, 2654, 86, 70, "DisplayMath",
 CellID->2004286477],
Cell[52509, 1583, 1031, 37, 70, "Text",
 CellID->1029232038],
Cell[53543, 1622, 718, 24, 70, "NumberedEquation",
 CellID->205611458],
Cell[54264, 1648, 758, 23, 70, "Text",
 CellID->1228512178],
Cell[55025, 1673, 488, 18, 70, "Text",
 CellID->942597820],
Cell[55516, 1693, 580, 19, 70, "DisplayMath",
 CellID->28632706],
Cell[56099, 1714, 76, 1, 70, "Text",
 CellID->321043162],
Cell[56178, 1717, 582, 19, 70, "DisplayMath",
 CellID->1348388825],
Cell[56763, 1738, 52, 1, 70, "Text",
 CellID->1283149602],
Cell[56818, 1741, 619, 20, 70, "DisplayMath",
 CellID->2093107450],
Cell[57440, 1763, 331, 11, 70, "Text",
 CellID->1398185347],
Cell[57774, 1776, 718, 21, 70, "NumberedEquation",
 CellID->99789938],
Cell[58495, 1799, 1293, 42, 70, "Text",
 CellID->93361577],
Cell[59791, 1843, 345, 10, 70, "Text",
 CellID->1076859350],
Cell[60139, 1855, 4078, 145, 70, "Text",
 CellID->431346174],
Cell[64220, 2002, 832, 29, 70, "DisplayMath",
 CellID->2074262724],
Cell[65055, 2033, 1208, 33, 70, "Text",
 CellID->868453947],
Cell[66266, 2068, 127, 3, 70, "MathCaption",
 CellID->1224517775],
Cell[CellGroupData[{
Cell[66418, 2075, 683, 21, 70, "Input",
 CellID->786868787],
Cell[67104, 2098, 467, 17, 36, "Output",
 CellID->4706927]
}, Open  ]],
Cell[67586, 2118, 77, 1, 70, "MathCaption",
 CellID->690415336],
Cell[CellGroupData[{
Cell[67688, 2123, 299, 10, 28, "Input",
 CellID->1697043452],
Cell[67990, 2135, 4338, 75, 251, "Output",
 Evaluatable->False,
 CellID->582922320]
}, Open  ]],
Cell[72343, 2213, 263, 8, 70, "MathCaption",
 CellID->953370135],
Cell[CellGroupData[{
Cell[72631, 2225, 2200, 62, 70, "Input",
 CellID->397105427],
Cell[74834, 2289, 773, 28, 54, "Output",
 CellID->58253176]
}, Open  ]],
Cell[75622, 2320, 187, 6, 70, "Text",
 CellID->1354278207],
Cell[75812, 2328, 116, 3, 70, "MathCaption",
 CellID->1345021535],
Cell[CellGroupData[{
Cell[75953, 2335, 146, 4, 70, "Input",
 CellID->407573496],
Cell[76102, 2341, 243, 8, 36, "Output",
 CellID->454740148]
}, Open  ]],
Cell[76360, 2352, 648, 20, 70, "Text",
 CellID->499104493],
Cell[77011, 2374, 117, 3, 70, "MathCaption",
 CellID->264376325],
Cell[CellGroupData[{
Cell[77153, 2381, 232, 7, 70, "Input",
 CellID->258011946],
Cell[77388, 2390, 350, 10, 39, "Output",
 CellID->31018780]
}, Open  ]],
Cell[77753, 2403, 447, 14, 70, "Text",
 CellID->1492888503],
Cell[78203, 2419, 98, 1, 70, "MathCaption",
 CellID->1200609766],
Cell[CellGroupData[{
Cell[78326, 2424, 658, 20, 70, "Input",
 CellID->733348657],
Cell[78987, 2446, 410, 8, 70, "Message",
 CellID->87486231],
Cell[79400, 2456, 484, 17, 36, "Output",
 CellID->645779371]
}, Open  ]],
Cell[79899, 2476, 138, 4, 70, "Text",
 CellID->2016498260],
Cell[80040, 2482, 173, 4, 70, "Text",
 CellID->1759555467],
Cell[80216, 2488, 314, 9, 70, "MathCaption",
 CellID->1799700291],
Cell[CellGroupData[{
Cell[80555, 2501, 780, 24, 70, "Input",
 CellID->2116288198],
Cell[81338, 2527, 398, 8, 70, "Message",
 CellID->332611952],
Cell[81739, 2537, 355, 12, 36, "Output",
 CellID->149070785]
}, Open  ]],
Cell[82109, 2552, 290, 8, 70, "MathCaption",
 CellID->400150724],
Cell[CellGroupData[{
Cell[82424, 2564, 950, 28, 70, "Input",
 CellID->1779696120],
Cell[83377, 2594, 411, 8, 70, "Message",
 CellID->456919939],
Cell[83791, 2604, 383, 14, 36, "Output",
 CellID->466855944]
}, Open  ]],
Cell[84189, 2621, 212, 5, 70, "MathCaption",
 CellID->1624668500],
Cell[CellGroupData[{
Cell[84426, 2630, 1167, 35, 101, "Input",
 CellID->419081260],
Cell[85596, 2667, 1878, 35, 224, "Output",
 Evaluatable->False,
 CellID->639613595]
}, Open  ]],
Cell[87489, 2705, 586, 21, 70, "Text",
 CellID->167600188],
Cell[88078, 2728, 1090, 28, 70, "DefinitionBox3Col",
 CellID->2116124712],
Cell[89171, 2758, 263, 10, 70, "Caption",
 CellID->1595931707],
Cell[89437, 2770, 142, 5, 70, "Text",
 CellID->1120970323],
Cell[89582, 2777, 897, 21, 70, "DefinitionBox3Col",
 CellID->365746491],
Cell[90482, 2800, 169, 7, 70, "Caption",
 CellID->1115642365],
Cell[90654, 2809, 210, 6, 70, "Text",
 CellID->327764883],
Cell[90867, 2817, 266, 9, 70, "MathCaption",
 CellID->1710768027],
Cell[CellGroupData[{
Cell[91158, 2830, 744, 23, 70, "Input",
 CellID->1523215861],
Cell[91905, 2855, 427, 15, 43, "Output",
 CellID->70995793]
}, Open  ]],
Cell[92347, 2873, 256, 9, 70, "MathCaption",
 CellID->62388872],
Cell[CellGroupData[{
Cell[92628, 2886, 1133, 34, 65, "Input",
 CellID->1787592186],
Cell[93764, 2922, 8807, 148, 221, "Output",
 Evaluatable->False,
 CellID->301569196]
}, Open  ]],
Cell[102586, 3073, 143, 4, 70, "MathCaption",
 CellID->767104824],
Cell[CellGroupData[{
Cell[102754, 3081, 1265, 37, 65, "Input",
 CellID->1132997041],
Cell[104022, 3120, 7856, 133, 221, "Output",
 Evaluatable->False,
 CellID->581476063]
}, Open  ]],
Cell[111893, 3256, 691, 21, 70, "Text",
 CellID->1500453481],
Cell[112587, 3279, 443, 14, 70, "MathCaption",
 CellID->799445058],
Cell[CellGroupData[{
Cell[113055, 3297, 1392, 40, 83, "Input",
 CellID->1416681447],
Cell[114450, 3339, 7652, 129, 223, "Output",
 Evaluatable->False,
 CellID->188794340]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[122151, 3474, 119, 2, 70, "Section",
 CellTags->{"c:3", "b:9.0", "ndsg:3.1.0"},
 CellID->3518691],
Cell[122273, 3478, 374, 7, 70, "Text",
 CellID->502458878],
Cell[122650, 3487, 89, 1, 70, "Text",
 CellID->153136180],
Cell[122742, 3490, 933, 34, 70, "Text",
 CellID->125679040],
Cell[123678, 3526, 252, 9, 70, "Text",
 CellID->66336125],
Cell[123933, 3537, 398, 16, 70, "Text",
 CellID->282084070],
Cell[124334, 3555, 529, 19, 70, "MathCaption",
 CellID->480930858],
Cell[CellGroupData[{
Cell[124888, 3578, 2146, 64, 101, "Input",
 CellID->753178145],
Cell[127037, 3644, 2470, 45, 158, "Output",
 Evaluatable->False,
 CellID->609963450]
}, Open  ]],
Cell[CellGroupData[{
Cell[129544, 3694, 71, 1, 70, "RelatedTutorialsSection",
 CellID->43850463],
Cell[129618, 3697, 326, 9, 70, "RelatedTutorials",
 CellID->604721514]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[129983, 3711, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

