(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    669713,      13735]
NotebookOptionsPosition[    640595,      12760]
NotebookOutlinePosition[    647326,      12963]
CellTagsIndexPosition[    646327,      12930]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Advanced Numerical Differential Equation Solving in \
Mathematica" :> 
         Documentation`HelpLookup["paclet:tutorial/NDSolveOverview"]}, #, 
        2]& , Evaluator -> Automatic], {
      1->"\<\"Advanced Numerical Differential Equation Solving in Mathematica\
\"\>"}, "\<\"related tutorials\"\>", 
      StyleBox["\<\"related tutorials\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["ExplicitRungeKutta Method for NDSolve", "Title",
 CellTags->"c:1",
 CellID->1812709110],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->689939648],

Cell["\<\
This loads packages containing some test problems and utility functions.\
\>", "Text",
 CellID->517979082],

Cell[BoxData[{
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveProblems`\>\"", "]"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveUtilities`\>\"", "]"}], 
  ";"}]}], "Input",
 InitializationCell->True,
 CellLabel->"In[3]:=",
 CellID->2013953080],

Cell[CellGroupData[{

Cell["Euler's method", "Subsection",
 CellTags->"ss:1",
 CellID->175801995],

Cell["\<\
One of the first and simplest methods for solving initial value problems was \
proposed by Euler:\
\>", "Text",
 CellID->2118371868],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["y", 
    RowBox[{"n", "+", "1"}]], " ", "=", " ", 
   RowBox[{
    SubscriptBox["y", 
     RowBox[{"n", " "}]], "+", " ", 
    RowBox[{"h", " ", 
     RowBox[{
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "n"], ",", " ", 
        SubscriptBox["y", "n"]}], ")"}], "."}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"EulersMethod",
 CellID->501353605],

Cell["Euler's method is not very accurate.", "Text",
 CellID->928734705],

Cell[TextData[{
 "Local accuracy is measured by how high terms are matched with the Taylor \
expansion of the solution. Euler's method is first-order accurate, so that \
errors occur one order higher starting at powers of ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    SuperscriptBox["h", "2"]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->701630704],

Cell[TextData[{
 "Euler's method is implemented in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " as ",
 StyleBox["ExplicitEuler", "MR"],
 "."
}], "Text",
 CellID->1718414918],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"-", 
       RowBox[{"y", "[", "t", "]"}]}]}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", 
   RowBox[{"y", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{"Method", "\[Rule]", "\"\<ExplicitEuler\>\""}], ",", " ", 
   RowBox[{"\"\<StartingStepSize\>\"", "\[Rule]", 
    RowBox[{"1", "/", "10"}]}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->1380692090],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
    RowBox[{
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False], "[", "t", "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{351, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->47958774]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Generalizing Euler's method", "Subsection",
 CellTags->"ss:2",
 CellID->863140840],

Cell["\<\
The idea of Runge-Kutta methods is to take successive (weighted) Euler steps \
to approximate a Taylor series. In this way function evaluations (and not \
derivatives) are used.\
\>", "Text",
 CellID->859415753],

Cell[TextData[{
 "For example, consider the one-step formulation of the ",
 StyleBox["midpoint method",
  FontSlant->"Italic"],
 ":"
}], "Text",
 CellID->748503971],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      SubscriptBox["k", "1"], " "}], "=", 
     RowBox[{" ", 
      RowBox[{"f", "(", 
       RowBox[{
        SubscriptBox["t", "n"], ",", " ", 
        SubscriptBox["y", "n"]}], ")"}]}]},
    {
     SubscriptBox["k", "2"], "=", 
     RowBox[{" ", 
      RowBox[{"f", "(", 
       RowBox[{
        RowBox[{
         SubscriptBox["t", "n"], "+", " ", 
         RowBox[{
          FractionBox["1", "2"], "h"}]}], ",", " ", 
        RowBox[{
         SubscriptBox["y", "n"], " ", "+", " ", 
         RowBox[{
          FractionBox["1", "2"], " ", "h", " ", 
          SubscriptBox["k", "1"]}]}]}], " ", ")"}]}]},
    {
     SubscriptBox["y", 
      RowBox[{"n", "+", "1"}]], "=", 
     RowBox[{" ", 
      RowBox[{
       SubscriptBox["y", "n"], " ", "+", " ", 
       RowBox[{"h", " ", 
        SubscriptBox["k", 
         RowBox[{"2", " "}]]}]}]}]}
   },
   GridBoxAlignment->{
    "Columns" -> {Center, Center, {Left}}, "ColumnsIndexed" -> {}, 
     "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"ExplicitMidpointMethod",
 CellID->1322871082],

Cell[TextData[{
 "The midpoint method can be shown to have a local error of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"O", "(", 
    SuperscriptBox["h", "3"], ")"}], TraditionalForm]]],
 " so",
 " ",
 "it is second-order accurate."
}], "Text",
 CellID->667234019],

Cell[TextData[{
 "The midpoint method is implemented in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " as ",
 StyleBox["ExplicitMidpoint", "MR"],
 "."
}], "Text",
 CellID->744826097],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{"-", 
       RowBox[{"y", "[", "t", "]"}]}]}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], "\[Equal]", "1"}]}], "}"}], ",", 
   RowBox[{"y", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{"Method", "\[Rule]", "\"\<ExplicitMidpoint\>\""}], ",", " ", 
   RowBox[{"\"\<StartingStepSize\>\"", "\[Rule]", 
    RowBox[{"1", "/", "10"}]}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->1017557097],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"y", "[", "t", "]"}], "\[Rule]", 
    RowBox[{
     TagBox[
      RowBox[{"InterpolatingFunction", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
       "]"}],
      False,
      Editable->False], "[", "t", "]"}]}], "}"}], "}"}]], "Output",
 ImageSize->{351, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->580441255]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Runge-Kutta methods", "Subsection",
 CellID->748768759],

Cell[TextData[{
 "Extending the approach in (",
 
 CounterBox["NumberedEquation", "ExplicitMidpointMethod"],
 "), repeated function evaluation can be used to obtain \
higher-and-higher-order methods."
}], "Text",
 CellID->1811365676],

Cell[TextData[{
 "Denote the Runge-Kutta method for the approximate solution to an initial \
value problem at ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{
     RowBox[{
      SubscriptBox["t", 
       RowBox[{"n", "+", "1"}]], " ", "=", " ", 
      RowBox[{
       SubscriptBox["t", "n"], " ", "+", " ", "h"}]}], ","}]}], 
   TraditionalForm]]],
 " ",
 "by:"
}], "Text",
 CellID->1305450463],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["g", "i"], 
     RowBox[{"=", " "}], 
     RowBox[{
      RowBox[{
       SubscriptBox["y", "n"], " ", "+", " ", 
       RowBox[{"h", " ", 
        RowBox[{
         UnderoverscriptBox["\[Sum]", 
          RowBox[{"j", "=", "1"}], "s"], " ", 
         RowBox[{
          SubscriptBox["a", 
           RowBox[{"i", ",", "j"}]], " ", 
          SubscriptBox["k", "j"]}]}]}]}], " ", ","}], " "},
    {
     SubscriptBox["k", "i"], "=", 
     RowBox[{
      RowBox[{"f", " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{
          SubscriptBox["t", "n"], " ", "+", " ", 
          RowBox[{
           SubscriptBox["c", "i"], " ", "h"}]}], ",", " ", 
         SubscriptBox["g", "i"]}], ")"}]}], ","}], 
     RowBox[{
      RowBox[{"i", " ", "=", " ", "1"}], ",", " ", "2", ",", "\[Ellipsis]", 
      ",", " ", "s", ","}]},
    {
     SubscriptBox["y", 
      RowBox[{"n", "+", "1"}]], "=", 
     RowBox[{
      SubscriptBox["y", "n"], " ", "+", " ", 
      RowBox[{"h", " ", 
       RowBox[{
        UnderoverscriptBox["\[Sum]", 
         RowBox[{"i", "=", "1"}], "s"], " ", 
        RowBox[{
         SubscriptBox["b", "i"], " ", 
         SubscriptBox["k", "i"], " "}]}]}]}], " "}
   },
   GridBoxAlignment->{
    "Columns" -> {Center, Center, Left, {Center}}, "ColumnsIndexed" -> {}, 
     "Rows" -> {{Baseline}}, "RowsIndexed" -> {}}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"ERKMethod",
 CellID->194574237],

Cell[TextData[{
 "where ",
 StyleBox["s",
  FontSlant->"Italic"],
 " is the number of stages."
}], "Text",
 CellID->1494356496],

Cell[TextData[{
 "It is generally assumed that the ",
 StyleBox["row-sum conditions",
  FontSlant->"Italic"],
 " hold:"
}], "Text",
 CellID->467022158],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["c", "i"], " ", "=", " ", 
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], "s"], " ", 
    SubscriptBox["a", 
     RowBox[{"i", ",", "j"}]]}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"RowSumConditions",
 CellID->1981247538],

Cell["\<\
These effectively determine the points in time at which the function is \
sampled and are a particularly useful device in the derivation of high-order \
Runge-Kutta methods.\
\>", "Text",
 CellID->1936944985],

Cell[TextData[{
 "The ",
 StyleBox["coefficients of the method ",
  FontSlant->"Italic"],
 "are ",
 StyleBox["free parameters",
  FontSlant->"Italic"],
 " that are chosen to satisfy a Taylor series expansion through some order in \
the time step ",
 StyleBox["h",
  FontSlant->"Italic"],
 ". In practice other conditions such as stability can also constrain the \
coefficients."
}], "Text",
 CellID->1512404568],

Cell[TextData[{
 "Explicit Runge-Kutta methods are a special case where the matrix ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", "A", " "}], TraditionalForm]]],
 " is strictly lower triangular:"
}], "Text",
 CellID->743953168],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["a", 
     RowBox[{"i", ",", " ", "j"}]], " ", "=", " ", "0"}], ",", " ", 
   RowBox[{"j", " ", "\[GreaterEqual]", "i"}], ",", " ", 
   RowBox[{"j", "=", " ", "1"}], ",", "\[Ellipsis]", ",", 
   RowBox[{"s", "."}]}], TraditionalForm]], "DisplayMath",
 CellID->341930292],

Cell[TextData[{
 "It has become customary to denote the method coefficients ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["c",
     FontWeight->"Bold"], " ", "=", " ", 
    FormBox[
     SuperscriptBox[
      RowBox[{"[", 
       SubscriptBox["c", "i"], "]"}], "T"],
     TraditionalForm]}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["b",
     FontWeight->"Bold"], "=", " ", 
    FormBox[
     SuperscriptBox[
      RowBox[{"[", 
       SubscriptBox["b", "i"], "]"}], "T"],
     TraditionalForm]}], TraditionalForm]]],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"A", " ", "=", " ", 
    RowBox[{"[", " ", 
     SubscriptBox["a", 
      RowBox[{"i", ",", "j"}]], " ", "]"}]}], TraditionalForm]]],
 " using a ",
 StyleBox["Butcher table",
  FontSlant->"Italic"],
 ", which has the following form for explicit Runge-Kutta methods:"
}], "Text",
 CellID->1250344399],

Cell[BoxData[
 FormBox[
  TagBox[GridBox[{
     {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
     {
      SubscriptBox["c", "2"], 
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]], "0", "\[CenterEllipsis]", "0", "0"},
     {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]", 
      "\[DescendingEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
     {
      SubscriptBox["c", "s"], 
      SubscriptBox["a", 
       RowBox[{"s", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"s", ",", "2"}]], "\[CenterEllipsis]", 
      SubscriptBox["a", 
       RowBox[{"s", ",", 
        RowBox[{"s", "-", "1"}]}]], "0"},
     {"", 
      SubscriptBox["b", "1"], 
      SubscriptBox["b", "2"], "\[CenterEllipsis]", 
      SubscriptBox["b", 
       RowBox[{"s", "-", "1"}]], 
      SubscriptBox["b", "s"]}
    },
    GridBoxDividers->{
     "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, False, False, False, {True}, False}, 
      "RowsIndexed" -> {}}],
   DisplayForm], TraditionalForm]], "NumberedEquation",
 CellTags->"ButcherTable",
 CellID->409178630],

Cell["\<\
The row-sum conditions can be visualized as summing across the rows of the \
table.\
\>", "Text",
 CellID->1031869940],

Cell[TextData[{
 "Notice that a consequence of explicitness is ",
 Cell[BoxData[
  FormBox[
   RowBox[{" ", 
    RowBox[{
     SubscriptBox["c", "1"], " ", "=", " ", "0"}]}], TraditionalForm]]],
 ", so that the function is sampled at the beginning of the current \
integration step."
}], "Text",
 CellID->1768323481]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example", "Subsection",
 CellID->275648062],

Cell[TextData[{
 "The Butcher table for the explicit midpoint method (",
 
 CounterBox["NumberedEquation", "ExplicitMidpointMethod"],
 ") is given by:"
}], "Text",
 CellID->1207671936],

Cell[BoxData[
 FormBox[
  TagBox[GridBox[{
     {"0", "0", "0"},
     {
      FractionBox["1", "2"], 
      FractionBox["1", "2"], "0"},
     {"", "0", "1"}
    },
    GridBoxDividers->{
     "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, False, {True}, False}, "RowsIndexed" -> {}}],
   DisplayForm], TraditionalForm]], "NumberedEquation",
 CellTags->"ExplicitMidpointButcherTable",
 CellID->1739211701]
}, Open  ]],

Cell[CellGroupData[{

Cell["FSAL schemes", "Subsection",
 CellID->777001667],

Cell["\<\
A particularly interesting special class of explicit Runge-Kutta methods, \
used in most modern codes, are those for which the coefficient have a special \
structure known as First Same As Last (FSAL):\
\>", "Text",
 CellID->2136406641],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["a", 
     RowBox[{"s", ",", "i"}]], " ", "=", " ", 
    SubscriptBox["b", "i"]}], ",", " ", 
   RowBox[{"i", " ", "=", " ", "1"}], ",", "\[Ellipsis]", ",", 
   RowBox[{
    RowBox[{"s", "-", 
     RowBox[{"1", " ", "and", " ", 
      SubscriptBox["b", "s"]}]}], "=", " ", "0."}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"FSALConditions",
 CellID->293358758],

Cell[TextData[{
 "For consistent FSAL schemes the Butcher table ",
 "(",
 
 CounterBox["NumberedEquation", "ButcherTable"],
 ") ",
 "has the form:"
}], "Text",
 CellID->347932937],

Cell[BoxData[
 FormBox[
  TagBox[GridBox[{
     {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
     {
      SubscriptBox["c", "2"], 
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]], "0", "\[CenterEllipsis]", "0", "0"},
     {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]", 
      "\[DescendingEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
     {
      SubscriptBox["c", 
       RowBox[{"s", "-", "1"}]], 
      SubscriptBox["a", 
       RowBox[{
        RowBox[{"s", "-", "1"}], ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{
        RowBox[{"s", "-", "1"}], ",", "2"}]], "\[CenterEllipsis]", "0", "0"},
     {"1", 
      SubscriptBox["b", "1"], 
      SubscriptBox["b", "2"], "\[CenterEllipsis]", 
      SubscriptBox["b", 
       RowBox[{"s", "-", "1"}]], "0"},
     {"", 
      SubscriptBox["b", "1"], 
      SubscriptBox["b", "2"], "\[CenterEllipsis]", 
      SubscriptBox["b", 
       RowBox[{"s", "-", "1"}]], "0"}
    },
    GridBoxDividers->{
     "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, False, False, False, False, {True}, False}, 
      "RowsIndexed" -> {}}],
   DisplayForm], TraditionalForm]], "NumberedEquation",
 CellTags->"FSALButcherTable",
 CellID->1369615083],

Cell[TextData[{
 "The advantage of FSAL methods is that the function value ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "s"], TraditionalForm]]],
 " at the end of one integration step is the same as the first function value \
",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "1"], TraditionalForm]]],
 " at the next integration step."
}], "Text",
 CellID->1156499598],

Cell[TextData[{
 "The function values at the beginning and end of each integration step are \
required anyway when constructing the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " that is used for dense output in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1207348691]
}, Open  ]],

Cell[CellGroupData[{

Cell["Embedded pairs and local error estimation", "Subsection",
 CellID->2118119054],

Cell[TextData[{
 "An efficient means of obtaining local error estimates for adaptive step \
size control is to consider two methods of different orders ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   OverscriptBox["p", "^"], TraditionalForm]]],
 " that share the same coefficient matrix (and hence function values)."
}], "Text",
 CellID->331887032],

Cell[BoxData[
 FormBox[
  TagBox[GridBox[{
     {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
     {
      SubscriptBox["c", "2"], 
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]], "0", "\[CenterEllipsis]", "0", "0"},
     {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]", 
      "\[DescendingEllipsis]", "0", "\[VerticalEllipsis]"},
     {
      SubscriptBox["c", 
       RowBox[{"s", "-", "1"}]], 
      SubscriptBox["a", 
       RowBox[{
        RowBox[{"s", "-", "1"}], ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{
        RowBox[{"s", "-", "1"}], ",", "2"}]], "\[CenterEllipsis]", "0", "0"},
     {
      SubscriptBox["c", "s"], 
      SubscriptBox["a", 
       RowBox[{"s", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"s", ",", "2"}]], "\[CenterEllipsis]", 
      SubscriptBox["a", 
       RowBox[{"s", ",", 
        RowBox[{"s", "-", "1"}]}]], "0"},
     {"", 
      SubscriptBox["b", "1"], 
      SubscriptBox["b", "2"], "\[CenterEllipsis]", 
      SubscriptBox["b", 
       RowBox[{"s", "-", "1"}]], 
      SubscriptBox["b", "s"]},
     {" ", 
      SubscriptBox[
       OverscriptBox["b", "^"], "1"], 
      SubscriptBox[
       OverscriptBox["b", "^"], "2"], "\[CenterEllipsis]", 
      SubscriptBox[
       OverscriptBox["b", "^"], 
       RowBox[{"s", "-", "1"}]], 
      SubscriptBox[
       OverscriptBox["b", "^"], "s"]}
    },
    GridBoxDividers->{
     "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, False, False, False, False, True, {False}, False}, 
      "RowsIndexed" -> {}}],
   DisplayForm], TraditionalForm]], "NumberedEquation",
 CellTags->"EmbeddedButcherTable",
 CellID->1874582710],

Cell["These give two solutions:", "Text",
 CellID->715461503],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["y", 
    RowBox[{"n", "+", "1"}]], " ", "=", " ", 
   RowBox[{
    SubscriptBox["y", "n"], " ", "+", " ", 
    RowBox[{"h", " ", 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "s"], " ", 
      RowBox[{
       SubscriptBox["b", "i"], " ", 
       SubscriptBox["k", "i"]}]}]}]}]}], TraditionalForm]], "NumberedEquation",\

 CellTags->"HigherOrder",
 CellID->1349444129],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox[
    OverscriptBox["y", "^"], 
    RowBox[{"n", "+", "1"}]], " ", "=", " ", 
   RowBox[{
    SubscriptBox["y", "n"], " ", "+", " ", 
    RowBox[{"h", " ", 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "s"], " ", 
      RowBox[{
       SubscriptBox[
        OverscriptBox["b", "^"], "i"], " ", 
       SubscriptBox["k", "i"]}]}]}]}]}], TraditionalForm]], "NumberedEquation",\

 CellTags->"LowerOrder",
 CellID->14497807],

Cell[TextData[{
 "A commonly used notation is ",
 Cell[BoxData[
  FormBox[
   RowBox[{"p", "(", 
    FormBox[
     OverscriptBox["p", "^"],
     TraditionalForm], ")"}], TraditionalForm]]],
 ", typically with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     OverscriptBox["p", "^"],
     TraditionalForm], "=", 
    RowBox[{"p", "-", "1"}]}], TraditionalForm]]],
 " or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     OverscriptBox["p", "^"],
     TraditionalForm], "=", 
    RowBox[{"p", "+", "1"}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1464230263],

Cell[TextData[{
 "In most modern codes, including the default choice in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ", the solution is advanced with the more accurate formula so that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     OverscriptBox["p", "^"],
     TraditionalForm], "=", 
    RowBox[{"p", "-", "1"}]}], TraditionalForm]]],
 ", which is known as local extrapolation."
}], "Text",
 CellID->1013326157],

Cell[TextData[{
 "The vector of coefficients ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    FormBox[
     RowBox[{"e", " ", "=", " ", 
      RowBox[{"[", 
       RowBox[{
        RowBox[{
         FormBox[
          SubscriptBox["b", "1"],
          TraditionalForm], "-", 
         FormBox[
          SubscriptBox[
           OverscriptBox["b", "^"], "1"],
          TraditionalForm]}], ",", " ", 
        RowBox[{
         FormBox[
          SubscriptBox["b", "2"],
          TraditionalForm], "-", 
         FormBox[
          SubscriptBox[
           OverscriptBox["b", "^"], "2"],
          TraditionalForm]}], ",", "\[Ellipsis]", ",", " ", 
        RowBox[{
         FormBox[
          SubscriptBox["b", "s"],
          TraditionalForm], "-", 
         FormBox[
          SubscriptBox[
           OverscriptBox["b", "^"], "s"],
          TraditionalForm]}]}], "]"}]}],
     TraditionalForm], "T"], TraditionalForm]]],
 " gives an error estimator avoiding subtractive cancellation of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "n"], TraditionalForm]]],
 " in floating-point arithmetic when forming the difference between (",
 
 CounterBox["NumberedEquation", "HigherOrder"],
 ") and (",
 
 CounterBox["NumberedEquation", "LowerOrder"],
 ")."
}], "Text",
 CellID->1319948270],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["err", "n"], " ", "=", " ", 
   RowBox[{"h", " ", 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"i", "=", "1"}], "s"], " ", 
     RowBox[{
      SubscriptBox["e", "i"], " ", 
      SubscriptBox["k", "i"]}]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->1658804500],

Cell[TextData[{
 "The quantity ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LeftDoubleBracketingBar]", " ", 
    SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}], 
   TraditionalForm]]],
 " gives a scalar measure of the error that can be used for step size \
selection."
}], "Text",
 CellID->360199005]
}, Open  ]],

Cell[CellGroupData[{

Cell["Step control", "Subsection",
 CellID->1786970428],

Cell["\<\
The classical, Integral or I step size controller uses the formula:\
\>", "Text",
 CellID->1492568390],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["h", 
    RowBox[{"n", "+", "1"}]], "=", " ", 
   RowBox[{
    SubscriptBox["h", "n"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox["Tol", 
       RowBox[{"\[LeftDoubleBracketingBar]", 
        SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}]], ")"}], 
     RowBox[{"1", "/", 
      OverscriptBox["p", "\[Tilde]"]}]]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"IController",
 CellID->149660343],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     OverscriptBox["p", "\[Tilde]"],
     TraditionalForm], " ", "=", " ", 
    RowBox[{
     FormBox[
      RowBox[{"min", "(", 
       RowBox[{
        OverscriptBox["p", "^"], ",", " ", "p"}], ")"}],
      TraditionalForm], " ", "+", " ", "1"}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1151891348],

Cell["\<\
The error estimate is therefore used to determine the next step size to use \
from the current step size.\
\>", "Text",
 CellID->89037911],

Cell[TextData[{
 "The notation ",
 Cell[BoxData[
  FormBox[
   RowBox[{"Tol", "/", 
    RowBox[{"\[LeftDoubleBracketingBar]", 
     SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}]}], 
   TraditionalForm]]],
 " is explained within \"",
 ButtonBox["Norms in NDSolve",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveVectorNorm"],
 "\"."
}], "Text",
 CellID->491590635]
}, Open  ]],

Cell[CellGroupData[{

Cell["Overview", "Subsection",
 CellID->254752099],

Cell["\<\
Explicit Runge Kutta pairs of orders 2(1) through 9(8) have been implemented.\
\
\>", "Text",
 CellID->498216541],

Cell["Formula pairs have the following properties:", "Text",
 CellID->1523460575],

Cell["First Same As Last strategy.", "BulletedText",
 CellID->2023075580],

Cell["\<\
Local extrapolation mode, that is the higher-order formula is used to \
propagate the solution.\
\>", "BulletedText",
 CellID->773864604],

Cell[TextData[{
 "Stiffness detection capability (see [",
 ButtonBox["HW96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#32197"],
 "], [",
 ButtonBox["P83",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2690"],
 "], [",
 ButtonBox["R87",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#28770"],
 "], [",
 ButtonBox["S84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#23832"],
 "])."
}], "BulletedText",
 CellID->583648863],

Cell[CellGroupData[{

Cell[TextData[{
 "Proportional-Integral step size controller for stiff and quasi-stiff \
systems [",
 ButtonBox["G91",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#26956"],
 "]."
}], "BulletedText",
 CellID->1614803634],

Cell[TextData[{
 "Optimal formula pairs of orders 2(1), 3(2), and 4(3) subject to the above \
requirements have been derived using ",
 StyleBox["Mathematica,",
  FontSlant->"Italic"],
 " and are described in [",
 ButtonBox["SS04",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#9420"],
 "]."
}], "Text",
 CellID->1550866271],

Cell[TextData[{
 "The 5(4) pair selected is due to Bogacki and Shampine [",
 ButtonBox["BS89b",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#4363"],
 ", ",
 ButtonBox["S94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#21224"],
 "] and the 6(5), 7(6), 8(7), and 9(8) pairs are due to Verner."
}], "Text",
 CellID->572571856],

Cell[TextData[{
 "For the selection of higher-order pairs, issues such as local truncation \
error ratio and stability region compatibility should be considered (see [",
 ButtonBox["S94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#21224"],
 "]) . Various tools have been written to assess these qualitative features."
}], "Text",
 CellID->1908162392],

Cell["\<\
Methods are interchangeable so that, for example, it is possible to \
substitute the 5(4) method of Bogacki and Shampine with a method of Dormand \
and Prince.\
\>", "Text",
 CellID->1898799209]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example", "Section",
 CellTags->"s:2",
 CellID->1942186939],

Cell["\<\
Define the Brusselator ODE problem, which models a chemical reaction.\
\>", "MathCaption",
 CellID->917178342],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"system", "=", 
  RowBox[{
  "GetNDSolveProblem", "[", "\"\<BrusselatorODE\>\"", "]"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->958664252],

Cell[BoxData[
 RowBox[{"NDSolveProblem", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "1"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{"1", "-", 
        RowBox[{"4", " ", 
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}]}], "+", 
        RowBox[{
         SuperscriptBox[
          RowBox[{
           SubscriptBox["Y", "1"], "[", "T", "]"}], "2"], " ", 
         RowBox[{
          SubscriptBox["Y", "2"], "[", "T", "]"}]}]}]}], ",", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         SubscriptBox["Y", "2"], "\[Prime]",
         MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
       RowBox[{
        RowBox[{"3", " ", 
         RowBox[{
          SubscriptBox["Y", "1"], "[", "T", "]"}]}], "-", 
        RowBox[{
         SuperscriptBox[
          RowBox[{
           SubscriptBox["Y", "1"], "[", "T", "]"}], "2"], " ", 
         RowBox[{
          SubscriptBox["Y", "2"], "[", "T", "]"}]}]}]}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "1"], "[", "0", "]"}], "\[Equal]", 
       FractionBox["3", "2"]}], ",", 
      RowBox[{
       RowBox[{
        SubscriptBox["Y", "2"], "[", "0", "]"}], "\[Equal]", "3"}]}], "}"}], 
    ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], ",", 
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"T", ",", "0", ",", "20"}], "}"}], ",", 
    RowBox[{"{", "}"}], ",", 
    RowBox[{"{", "}"}], ",", 
    RowBox[{"{", "}"}]}], "}"}], "]"}]], "Output",
 ImageSize->{540, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->489867450]
}, Open  ]],

Cell["This solves the system using an explicit Runge-Kutta method.", \
"MathCaption",
 CellID->1249065375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"sol", " ", "=", " ", 
  RowBox[{"NDSolve", "[", 
   RowBox[{"system", ",", 
    RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[8]:=",
 CellID->685763314],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->57136273]
}, Open  ]],

Cell["Extract the interpolating functions from the solution.", "MathCaption",
 CellID->451025856],

Cell[BoxData[
 RowBox[{
  RowBox[{"ifuns", " ", "=", " ", 
   RowBox[{
    RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}], " ", "/.", 
    RowBox[{"First", "[", "sol", "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->700261121],

Cell["Plot the solution components.", "MathCaption",
 CellID->1945030464],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", "ifuns", "]"}], ",", 
   RowBox[{"Evaluate", "[", 
    RowBox[{"system", "[", "\"\<TimeData\>\"", "]"}], "]"}]}], "]"}]], "Input",\

 CellLabel->"In[10]:=",
 CellID->1464352014],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 209},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->32302612]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Method comparison", "Section",
 CellTags->"s:3",
 CellID->1456351317],

Cell[TextData[{
 "Sometimes you may be interested to find out what methods are being used in \
",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1824551019],

Cell["\<\
Here you can see the coefficients of the default 2(1) embedded pair.\
\>", "MathCaption",
 CellID->1659696171],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve`EmbeddedExplicitRungeKuttaCoefficients", "[", 
  RowBox[{"2", ",", "Infinity"}], "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->801436059],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       FractionBox["1", "2"], ",", 
       FractionBox["1", "2"]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", "2"], ",", 
     FractionBox["1", "2"], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", "2"]}], ",", 
     FractionBox["2", "3"], ",", 
     RowBox[{"-", 
      FractionBox["1", "6"]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{336, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->635475014]
}, Open  ]],

Cell["\<\
Another issue is that you may want to compare some of the different methods \
to see how they perform for a specific problem.\
\>", "Text",
 CellID->664103872],

Cell[CellGroupData[{

Cell["Utilities", "Subsection",
 CellID->890473970],

Cell[TextData[{
 "We will make use of a utility function ",
 StyleBox["CompareMethods", "MR"],
 " for comparing various methods. Some useful ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " features of this function for comparing methods are:"
}], "Text",
 CellID->318318349],

Cell[TextData[{
 "The option ",
 StyleBox["EvaluationMonitor", "MR"],
 ", which is used to count the number of function evaluations."
}], "BulletedText",
 CellID->918855971],

Cell[CellGroupData[{

Cell[TextData[{
 "The option ",
 StyleBox["StepMonitor", "MR"],
 ", which is used to count the number of accepted and rejected integration \
steps."
}], "BulletedText",
 CellID->1388690512],

Cell[TextData[{
 "This displays the results of the method comparison using a ",
 StyleBox["GridBox", "MR"],
 "."
}], "MathCaption",
 CellID->1841778925],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"TabulateResults", "[", 
    RowBox[{"labels_List", ",", "names_List", ",", "data_List"}], "]"}], ":=",
    "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"DisplayForm", "[", "\[IndentingNewLine]", 
     RowBox[{"FrameBox", "[", "\[IndentingNewLine]", 
      RowBox[{"GridBox", "[", "\[IndentingNewLine]", 
       RowBox[{
        RowBox[{"Apply", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{"labels", ",", "##"}], "}"}], "&"}], ",", 
          RowBox[{"MapThread", "[", 
           RowBox[{"Prepend", ",", 
            RowBox[{"{", 
             RowBox[{"data", ",", "names"}], "}"}]}], "]"}]}], "]"}], ",", 
        "\[IndentingNewLine]", 
        RowBox[{"RowLines", "\[Rule]", "True"}], ",", 
        RowBox[{"ColumnLines", "\[Rule]", "True"}]}], "\[IndentingNewLine]", 
       "]"}], "\[IndentingNewLine]", "]"}], "\[IndentingNewLine]", "]"}], " ",
     "/;", " ", 
    RowBox[{"SameQ", "[", 
     RowBox[{
      RowBox[{"Length", "[", "names", "]"}], ",", " ", 
      RowBox[{"Length", "[", "data", "]"}]}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->794447904]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reference solution", "Subsection",
 CellID->15071468],

Cell["\<\
A number of examples for comparing numerical methods in the literature rely \
on the fact that a closed-form solution is available, which is obviously \
quite limiting.\
\>", "Text",
 CellID->277944140],

Cell[TextData[{
 "In ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " it is possible to get very accurate approximations using \
arbitrary-precision adaptive step size, adaptive order methods based on ",
 StyleBox["Extrapolation", "MR"],
 "."
}], "Text",
 CellID->261767890],

Cell[TextData[{
 "The following reference solution is computed with a method that switches \
between a pair of ",
 StyleBox["Extrapolation", "MR"],
 " methods, depending on whether the problem appears to be stiff."
}], "MathCaption",
 CellID->1565937739],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"sol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", "\"\<StiffnessSwitching\>\""}], ",", 
      RowBox[{"WorkingPrecision", "\[Rule]", "32"}]}], "]"}]}], ";"}], 
  "\n"}], "\n", 
 RowBox[{
  RowBox[{"refsol", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"FinalSolutions", "[", 
     RowBox[{"system", ",", "sol"}], "]"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[13]:=",
 CellID->850515997]
}, Open  ]],

Cell[CellGroupData[{

Cell["Automatic order selection", "Subsection",
 CellID->22446634],

Cell[TextData[{
 "When you select ",
 StyleBox["DifferenceOrder\[Rule]Automatic", "MR"],
 ", the code will automatically attempt to choose the optimal order method \
for the integration."
}], "Text",
 CellID->1317253300],

Cell[TextData[{
 "Two algorithms have been implemented for this purpose and are described \
within ",
 StyleBox[ButtonBox["SymplecticPartitionedRungeKutta",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveSPRK"], "MR"],
 "."
}], "Text",
 CellID->645219288]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example 1", "Subsection",
 CellID->939624976],

Cell["\<\
Here is an example that compares built-in methods of various orders, together \
with the method that is selected automatically.\
\>", "Text",
 CellID->128433904],

Cell[TextData[{
 "This selects the order of the methods to choose between and makes a list of \
method options to pass to ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->410557986],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"orders", " ", "=", " ", 
    RowBox[{"Join", "[", 
     RowBox[{
      RowBox[{"Range", "[", 
       RowBox[{"2", ",", "9"}], "]"}], ",", 
      RowBox[{"{", "Automatic", "}"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{"methods", " ", "=", " ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", 
        RowBox[{"Part", "[", 
         RowBox[{"orders", ",", "i"}], "]"}]}], ",", " ", 
       RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", " ", 
       RowBox[{"Length", "[", "orders", "]"}]}], "}"}]}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[15]:=",
 CellID->41206703],

Cell["\<\
Compute the number of integration steps, function evaluations, and the \
endpoint global error.\
\>", "MathCaption",
 CellID->814197219],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"CompareMethods", "[", 
    RowBox[{"system", ",", " ", "refsol", ",", " ", "methods"}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->891358188],

Cell["Display the results in a table.", "MathCaption",
 CellID->492232560],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"labels", " ", "=", " ", 
    RowBox[{"{", 
     RowBox[{
     "\"\<Method\>\"", ",", "\"\<Steps\>\"", ",", "\"\<Cost\>\"", ",", 
      "\"\<Error\>\""}], "}"}]}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{"TabulateResults", "[", 
  RowBox[{"labels", ",", " ", "orders", ",", " ", "data"}], "]"}]}], "Input",
 CellLabel->"In[18]:=",
 CellID->1514458340],

Cell[BoxData[
 TagBox[
  FrameBox[GridBox[{
     {"Method", "Steps", "Cost", "Error"},
     {"2", 
      RowBox[{"{", 
       RowBox[{"124381", ",", "2"}], "}"}], "248768", 
      "1.906816232688513`*^-8"},
     {"3", 
      RowBox[{"{", 
       RowBox[{"4247", ",", "1"}], "}"}], "12746", 
      "3.4670483206600267`*^-8"},
     {"4", 
      RowBox[{"{", 
       RowBox[{"940", ",", "6"}], "}"}], "3786", "9.306991510804608`*^-9"},
     {"5", 
      RowBox[{"{", 
       RowBox[{"188", ",", "17"}], "}"}], "1437", "6.091401383301331`*^-9"},
     {"6", 
      RowBox[{"{", 
       RowBox[{"290", ",", "15"}], "}"}], "2442", 
      "1.4619949695315881`*^-10"},
     {"7", 
      RowBox[{"{", 
       RowBox[{"165", ",", "16"}], "}"}], "1812", "8.853158206534317`*^-10"},
     {"8", 
      RowBox[{"{", 
       RowBox[{"86", ",", "14"}], "}"}], "1302", "1.452407705215819`*^-8"},
     {"9", 
      RowBox[{"{", 
       RowBox[{"90", ",", "21"}], "}"}], "1778", "1.2246804873910579`*^-8"},
     {"Automatic", 
      RowBox[{"{", 
       RowBox[{"90", ",", "21"}], "}"}], "1779", "1.2246804873910579`*^-8"}
    },
    GridBoxDividers->{
     "Columns" -> {False, {True}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, {True}, False}, "RowsIndexed" -> {}}]],
  DisplayForm]], "Output",
 ImageSize->{316, 193},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]//DisplayForm=",
 CellID->182305699]
}, Open  ]],

Cell["\<\
The default method has order nine, which is close to the optimal order of \
eight in this example. One function evaluation is needed during the \
initialization phase to determine the order.\
\>", "Text",
 CellID->1956584574]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example 2", "Subsection",
 CellID->1387184084],

Cell["\<\
A limitation of the previous example is that it did not take into account the \
accuracy of the solution obtained by each method, so that it did not give a \
fair reflection of the cost.\
\>", "Text",
 CellID->2145256321],

Cell["\<\
Rather than taking a single tolerance to compare methods, it is preferable to \
use a range of tolerances.\
\>", "Text",
 CellID->1633651945],

Cell[TextData[{
 "The following example compares various ",
 StyleBox["ExplicitRungeKutta", "MR"],
 " methods of different orders using a variety of tolerances."
}], "Text",
 CellID->899396607],

Cell[TextData[{
 "This selects the order of the methods to choose between and makes a list of \
method options to pass to ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->799208733],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"orders", "=", 
    RowBox[{"Join", "[", 
     RowBox[{
      RowBox[{"Range", "[", 
       RowBox[{"4", ",", "9"}], "]"}], ",", 
      RowBox[{"{", "Automatic", "}"}]}], "]"}]}], ";"}], "\n"}], "\n", 
 RowBox[{
  RowBox[{"methods", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", 
        RowBox[{"Part", "[", 
         RowBox[{"orders", ",", "i"}], "]"}]}], ",", 
       RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", 
       RowBox[{"Length", "[", "orders", "]"}]}], "}"}]}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[20]:=",
 CellID->1460379660],

Cell[TextData[{
 "The data comparing accuracy and work is computed using ",
 StyleBox["CompareMethods", "MR"],
 " for a range of tolerances."
}], "MathCaption",
 CellID->1563500479],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"Map", "[", 
      RowBox[{"Rest", ",", 
       RowBox[{"CompareMethods", "[", 
        RowBox[{"system", ",", "refsol", ",", "methods", ",", 
         RowBox[{"AccuracyGoal", "\[Rule]", "tol"}], ",", 
         RowBox[{"PrecisionGoal", "\[Rule]", "tol"}]}], "]"}]}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"tol", ",", "3", ",", "14"}], "}"}]}], "]"}]}], ";"}]], "Input",\

 CellLabel->"In[22]:=",
 CellID->1689682703],

Cell["\<\
The work-error comparison data for the various methods is displayed in the \
following logarithmic plot, where the global error is displayed on the \
vertical axis and the number of function evaluations on the horizontal axis. \
The default order selected (displayed in red) can be seen to increase as the \
tolerances are increased.\
\>", "MathCaption",
 CellID->930889567],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLogLogPlot", "[", 
  RowBox[{
   RowBox[{"Transpose", "[", "data", "]"}], ",", 
   RowBox[{"Joined", "\[Rule]", "True"}], ",", 
   RowBox[{"Axes", "\[Rule]", "False"}], ",", 
   RowBox[{"Frame", "\[Rule]", "True"}], ",", 
   RowBox[{"PlotMarkers", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", "Medium"}], "]"}], "&"}], ",", 
      RowBox[{"Join", "[", 
       RowBox[{
        RowBox[{"Drop", "[", 
         RowBox[{"orders", ",", 
          RowBox[{"-", "1"}]}], "]"}], ",", 
        RowBox[{"{", "\"\<A\>\"", "}"}]}], "]"}]}], "]"}]}], ",", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "Black", "}"}], ",", 
      RowBox[{"{", "Black", "}"}], ",", 
      RowBox[{"{", "Black", "}"}], ",", 
      RowBox[{"{", "Black", "}"}], ",", 
      RowBox[{"{", "Black", "}"}], ",", 
      RowBox[{"{", "Black", "}"}], ",", 
      RowBox[{"{", "Red", "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[23]:=",
 CellID->1472993286],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 120},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->395729241]
}, Open  ]],

Cell["\<\
The order selection algorithms are heuristic in that the optimal order may \
change through the integration but, as the above examples illustrate, a \
reasonable default choice is usually made.\
\>", "Text",
 CellID->945961035],

Cell["\<\
Ideally, a selection of different problems should be used for benchmarking.\
\>", "Text",
 CellID->1750937427]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Coefficient plug-in", "Section",
 CellTags->"s:4",
 CellID->1391088597],

Cell[TextData[{
 "The implementation of ",
 StyleBox["ExplicitRungeKutta", "MR"],
 " provides a default method pair at each order."
}], "Text",
 CellID->364415127],

Cell["\<\
Sometimes, however, it is convenient to use a different method, for example:\
\>", "Text",
 CellID->453969776],

Cell["To replicate the results of someone else.", "BulletedText",
 CellID->1413535231],

Cell["\<\
To use a special-purpose method that works well for a specific problem.\
\>", "BulletedText",
 CellID->1681668427],

Cell["To experiment with a new method", "BulletedText",
 CellID->1099189064],

Cell[CellGroupData[{

Cell["The classical Runge-Kutta method", "Subsection",
 CellID->205676559],

Cell[TextData[{
 "This shows how to define the coefficients of the classical explicit \
Runge-Kutta method of order four, approximated to precision ",
 StyleBox["p", "MR"],
 "."
}], "MathCaption",
 CellID->1810102332],

Cell[BoxData[
 RowBox[{
  RowBox[{"crkamat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", "/", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"1", "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}], ";", 
  RowBox[{"crkbvec", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "6"}], ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"1", "/", "6"}]}], "}"}]}], ";", 
  RowBox[{"crkcvec", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "2"}], ",", 
     RowBox[{"1", "/", "2"}], ",", "1"}], "}"}]}], ";", 
  RowBox[{
   RowBox[{"ClassicalRungeKuttaCoefficients", "[", 
    RowBox[{"4", ",", "p_"}], "]"}], ":=", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"crkamat", ",", "crkbvec", ",", "crkcvec"}], "}"}], ",", "p"}], 
    "]"}]}], ";"}]], "Input",
 CellLabel->"In[24]:=",
 CellID->1509747051],

Cell["\<\
The method has no embedded error estimate and hence there is no specification \
of the coefficient error vector. This means that the method is invoked with \
fixed step sizes.\
\>", "Text",
 CellID->246883263],

Cell["Here is an example of the calling syntax.", "MathCaption",
 CellID->476815903],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{"system", ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
      RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "4"}], ",", 
      RowBox[{
      "\"\<Coefficients\>\"", "\[Rule]", 
       "ClassicalRungeKuttaCoefficients"}]}], "}"}]}], ",", 
   RowBox[{"StartingStepSize", "\[Rule]", 
    RowBox[{"1", "/", "10"}]}]}], "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->253621591],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->59681361]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["ode23", "Subsection",
 CellID->1991274243],

Cell["\<\
This defines the coefficients for a 3(2) FSAL explicit Runge-Kutta pair.\
\>", "Text",
 CellID->942526595],

Cell[TextData[{
 "The third-order formula is due to Ralston, and the embedded method was \
derived by Bogacki and Shampine [",
 ButtonBox["BS89a",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2609"],
 "]."
}], "Text",
 CellID->548216749],

Cell["\<\
This defines function for computing the coefficients to a desired precision.\
\>", "MathCaption",
 CellID->1230974679],

Cell[BoxData[{
 RowBox[{
  RowBox[{"BSamat", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", "/", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", 
       RowBox[{"3", "/", "4"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"2", "/", "9"}], ",", 
       RowBox[{"1", "/", "3"}], ",", 
       RowBox[{"4", "/", "9"}]}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"BSbvec", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"2", "/", "9"}], ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"4", "/", "9"}], ",", "0"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"BScvec", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "2"}], ",", 
     RowBox[{"3", "/", "4"}], ",", "1"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"BSevec", "=", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"-", "5"}], "/", "72"}], ",", 
      RowBox[{"1", "/", "12"}], ",", 
      RowBox[{"1", "/", "9"}], ",", 
      RowBox[{
       RowBox[{"-", "1"}], "/", "8"}]}], "}"}]}], ";"}], "\n"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"BSCoefficients", "[", 
    RowBox[{"4", ",", " ", "p_"}], "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"BSamat", ",", "BSbvec", ",", "BScvec", ",", "BSevec"}], "}"}], 
     ",", " ", "p"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[26]:=",
 CellID->940299269],

Cell[TextData[{
 "The method is used in the Texas Instruments TI-85 pocket calculator, Matlab \
and RKSUITE [",
 ButtonBox["S94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#21224"],
 "]."
}], "Text",
 CellID->34385046],

Cell["\<\
Unfortunately it does not allow for the form of stiffness detection that has \
been chosen.\
\>", "Text",
 CellID->2130371287]
}, Open  ]],

Cell[CellGroupData[{

Cell["A method of Fehlberg", "Subsection",
 CellID->1739508002],

Cell[TextData[{
 "This defines the coefficients for a 4(5) explicit Runge-Kutta pair of \
Fehlberg that was popular in the 1960s [",
 ButtonBox["F69",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#145"],
 "]."
}], "Text",
 CellID->1494764706],

Cell["\<\
The fourth order formula is used to propagate the solution and the \
fifth-order formula is used only for the purpose of error estimation.\
\>", "Text",
 CellID->1546388118],

Cell["\<\
This defines function for computing the coefficients to a desired precision.\
\>", "MathCaption",
 CellID->1281707057],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Fehlbergamat", "=", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", "/", "4"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"3", "/", "32"}], ",", 
       RowBox[{"9", "/", "32"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"1932", "/", "2197"}], ",", 
       RowBox[{
        RowBox[{"-", "7200"}], "/", "2197"}], ",", 
       RowBox[{"7296", "/", "2197"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"439", "/", "216"}], ",", 
       RowBox[{"-", "8"}], ",", 
       RowBox[{"3680", "/", "513"}], ",", 
       RowBox[{
        RowBox[{"-", "845"}], "/", "4104"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"-", "8"}], "/", "27"}], ",", "2", ",", 
       RowBox[{
        RowBox[{"-", "3544"}], "/", "2565"}], ",", 
       RowBox[{"1859", "/", "4104"}], ",", 
       RowBox[{
        RowBox[{"-", "11"}], "/", "40"}]}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Fehlbergbvec", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"25", "/", "216"}], ",", "0", ",", 
     RowBox[{"1408", "/", "2565"}], ",", 
     RowBox[{"2197", "/", "4104"}], ",", 
     RowBox[{
      RowBox[{"-", "1"}], "/", "5"}], ",", " ", "0"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Fehlbergcvec", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "4"}], ",", 
     RowBox[{"3", "/", "8"}], ",", 
     RowBox[{"12", "/", "13"}], ",", "1", ",", 
     RowBox[{"1", "/", "2"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Fehlbergevec", "=", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"-", "1"}], "/", "360"}], ",", "0", ",", 
      RowBox[{"128", "/", "4275"}], ",", 
      RowBox[{"2197", "/", "75240"}], ",", 
      RowBox[{
       RowBox[{"-", "1"}], "/", "50"}], ",", 
      RowBox[{
       RowBox[{"-", "2"}], "/", "55"}]}], "}"}]}], ";"}], "\n"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"FehlbergCoefficients", "[", 
    RowBox[{"4", ",", " ", "p_"}], "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "Fehlbergamat", ",", "Fehlbergbvec", ",", "Fehlbergcvec", ",", 
       "Fehlbergevec"}], "}"}], ",", " ", "p"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[31]:=",
 CellID->1125979068],

Cell["\<\
In contrast to the classical Runge-Kutta method of order four, the \
coefficients include an additional entry that is used for error estimation.\
\>", "Text",
 CellID->1881260921],

Cell[TextData[{
 "The Fehlberg method is not a FSAL scheme since the coefficient matrix is \
not of the form (",
 
 CounterBox["NumberedEquation", "FSALButcherTable"],
 "); it is a six-stage scheme, but it requires six function evaluations per \
step because of the function evaluation that is required at the end of the \
step to construct the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 "."
}], "Text",
 CellID->65112977]
}, Open  ]],

Cell[CellGroupData[{

Cell["A DOrmand-PRInce method", "Subsection",
 CellID->451488706],

Cell[TextData[{
 "Here is how to define a 5(4) pair of Dormand and Prince [",
 ButtonBox["DP80",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#20618"],
 "]. This is currently the method used by ",
 StyleBox["ode45", "MR"],
 " in Matlab."
}], "Text",
 CellID->863097980],

Cell["\<\
This defines function for computing the coefficients to a desired precision.\
\>", "MathCaption",
 CellID->1011074012],

Cell[BoxData[{
 RowBox[{
  RowBox[{"DOPRIamat", "=", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", "/", "5"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"3", "/", "40"}], ",", 
       RowBox[{"9", "/", "40"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"44", "/", "45"}], ",", 
       RowBox[{
        RowBox[{"-", "56"}], "/", "15"}], ",", 
       RowBox[{"32", "/", "9"}]}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"19372", "/", "6561"}], ",", 
       RowBox[{
        RowBox[{"-", "25360"}], "/", "2187"}], ",", 
       RowBox[{"64448", "/", "6561"}], ",", 
       RowBox[{
        RowBox[{"-", "212"}], "/", "729"}]}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"9017", "/", "3168"}], ",", 
       RowBox[{
        RowBox[{"-", "355"}], "/", "33"}], ",", 
       RowBox[{"46732", "/", "5247"}], ",", 
       RowBox[{"49", "/", "176"}], ",", 
       RowBox[{
        RowBox[{"-", "5103"}], "/", "18656"}]}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"35", "/", "384"}], ",", "0", ",", 
       RowBox[{"500", "/", "1113"}], ",", 
       RowBox[{"125", "/", "192"}], ",", 
       RowBox[{
        RowBox[{"-", "2187"}], "/", "6784"}], ",", 
       RowBox[{"11", "/", "84"}]}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"DOPRIbvec", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"35", "/", "384"}], ",", "0", ",", 
     RowBox[{"500", "/", "1113"}], ",", 
     RowBox[{"125", "/", "192"}], ",", 
     RowBox[{
      RowBox[{"-", "2187"}], "/", "6784"}], ",", 
     RowBox[{"11", "/", "84"}], ",", " ", "0"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"DOPRIcvec", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"1", "/", "5"}], ",", 
     RowBox[{"3", "/", "10"}], ",", 
     RowBox[{"4", "/", "5"}], ",", 
     RowBox[{"8", "/", "9"}], ",", "1", ",", " ", "1"}], "}"}]}], ";"}], "\n",
 
 RowBox[{
  RowBox[{
   RowBox[{"DOPRIevec", "=", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"71", "/", "57600"}], ",", "0", ",", 
      RowBox[{
       RowBox[{"-", "71"}], "/", "16695"}], ",", 
      RowBox[{"71", "/", "1920"}], ",", 
      RowBox[{
       RowBox[{"-", "17253"}], "/", "339200"}], ",", 
      RowBox[{"22", "/", "525"}], ",", 
      RowBox[{
       RowBox[{"-", "1"}], "/", "40"}]}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"DOPRICoefficients", "[", 
    RowBox[{"5", ",", " ", "p_"}], "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"N", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "DOPRIamat", ",", "DOPRIbvec", ",", "DOPRIcvec", ",", "DOPRIevec"}], 
      "}"}], ",", " ", "p"}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[36]:=",
 CellID->54942485],

Cell[TextData[{
 "The Dormand-Prince method is a FSAL scheme since the coefficient matrix is \
of the form (",
 
 CounterBox["NumberedEquation", "FSALButcherTable"],
 "); it is a seven-stage scheme, but effectively uses only six function \
evaluations."
}], "Text",
 CellID->106702929],

Cell["\<\
Here is how the coefficients of Dormand and Princeton be used in place of the \
built-in choice. Since the structure of the coefficients includes an error \
vector, the implementation is able to ascertain that adaptive step sizes can \
be computed.\
\>", "MathCaption",
 CellID->1375489139],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{"system", ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
      RowBox[{"DifferenceOrder", "\[Rule]", "5"}], ",", 
      RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "DOPRICoefficients"}], ",", 
      " ", 
      RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[41]:=",
 CellID->508912261],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->26032513]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Method comparison", "Subsection",
 CellID->909673919],

Cell["\<\
Here we solve a system using several explicit Runge-Kutta pairs.\
\>", "Text",
 CellID->1608101326],

Cell[TextData[{
 "For the Fehlberg 4(5) pair, the option ",
 StyleBox["EmbeddedDifferenceOrder", "MR"],
 " is used to specify the order of the embedded method."
}], "MathCaption",
 CellID->1725914720],

Cell[BoxData[
 RowBox[{
  RowBox[{"Fehlberg45", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
     RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "FehlbergCoefficients"}], ",",
      
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "4"}], ",", 
     RowBox[{"\"\<EmbeddedDifferenceOrder\>\"", "\[Rule]", "5"}], ",", " ", 
     RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[42]:=",
 CellID->572292603],

Cell["The Dormand and Prince 5(4) pair is defined as follows.", "MathCaption",
 CellID->970364729],

Cell[BoxData[
 RowBox[{
  RowBox[{"DOPRI54", " ", "=", 
   RowBox[{"{", 
    RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
     RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "DOPRICoefficients"}], ",", 
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "5"}], ",", " ", 
     RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[43]:=",
 CellID->1972797983],

Cell["\<\
The 5(4) pair of Bogacki and Shampine is the default order-five method.\
\>", "MathCaption",
 CellID->1049108506],

Cell[BoxData[
 RowBox[{
  RowBox[{"BS54", " ", "=", 
   RowBox[{"{", 
    RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
     RowBox[{
     "\"\<Coefficients\>\"", "\[Rule]", 
      "\"\<EmbeddedExplicitRungeKuttaCoefficients\>\""}], ",", 
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "5"}], ",", " ", 
     RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[44]:=",
 CellID->1223986320],

Cell["\<\
Put the methods and some descriptive names together in a list.\
\>", "MathCaption",
 CellID->564430158],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"names", " ", "=", " ", 
    RowBox[{"{", 
     RowBox[{
     "\"\<Fehlberg 4(5)\>\"", ",", " ", "\"\<Dormand-Prince 5(4)\>\"", ",", 
      "\"\<Bogacki-Shampine 5(4)\>\""}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{"methods", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"Fehlberg45", ",", "DOPRI54", ",", "BS54"}], "}"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[45]:=",
 CellID->892899101],

Cell["\<\
Compute the number of integration steps, function evaluations, and the \
endpoint global error.\
\>", "MathCaption",
 CellID->19029267],

Cell[BoxData[
 RowBox[{
  RowBox[{"data", " ", "=", " ", 
   RowBox[{"CompareMethods", "[", 
    RowBox[{"system", ",", "refsol", ",", " ", "methods"}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[47]:=",
 CellID->1112646907],

Cell["Display the results in a table.", "MathCaption",
 CellID->2123873781],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"labels", " ", "=", " ", 
    RowBox[{"{", 
     RowBox[{
     "\"\<Method\>\"", ",", "\"\<Steps\>\"", ",", "\"\<Cost\>\"", ",", 
      "\"\<Error\>\""}], "}"}]}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{"TabulateResults", "[", 
  RowBox[{"labels", ",", " ", "names", ",", " ", "data"}], "]"}]}], "Input",
 CellLabel->"In[48]:=",
 CellID->959328536],

Cell[BoxData[
 TagBox[
  FrameBox[GridBox[{
     {"Method", "Steps", "Cost", "Error"},
     {
      RowBox[{"Fehlberg", " ", "4", 
       RowBox[{"(", "5", ")"}]}], 
      RowBox[{"{", 
       RowBox[{"320", ",", "12"}], "}"}], "1982", "1.6590419704698434`*^-7"},
     {
      RowBox[{"Dormand", "-", 
       RowBox[{"Prince", " ", "5", 
        RowBox[{"(", "4", ")"}]}]}], 
      RowBox[{"{", 
       RowBox[{"292", ",", "10"}], "}"}], "1814", "1.7579946498536003`*^-8"},
     {
      RowBox[{"Bogacki", "-", 
       RowBox[{"Shampine", " ", "5", 
        RowBox[{"(", "4", ")"}]}]}], 
      RowBox[{"{", 
       RowBox[{"188", ",", "17"}], "}"}], "1437", "6.091401383301331`*^-9"}
    },
    GridBoxDividers->{
     "Columns" -> {False, {True}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, {True}, False}, "RowsIndexed" -> {}}]],
  DisplayForm]], "Output",
 ImageSize->{367, 79},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[49]//DisplayForm=",
 CellID->314363219]
}, Open  ]],

Cell["\<\
The default method was the least expensive and provided the most accurate \
solution.\
\>", "Text",
 CellID->2106761420]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Method plug-in", "Section",
 CellTags->"s:5",
 CellID->551352890],

Cell["\<\
This shows how to implement the classical explicit Runge-Kutta method of \
order four using the method plug-in environment.\
\>", "Text",
 CellID->494313012],

Cell["\<\
This definition is optional since the method in fact has no data. However, \
any expression can be stored inside the data object. For example, the \
coefficients could be approximated here to avoid coercion from rational to \
floating-point numbers at each integration step.\
\>", "MathCaption",
 CellID->1505665890],

Cell[BoxData[
 RowBox[{
  RowBox[{"ClassicalRungeKutta", "/:", 
   RowBox[{"NDSolve`InitializeMethod", "[", 
    RowBox[{"ClassicalRungeKutta", ",", "__"}], "]"}], ":=", 
   RowBox[{"ClassicalRungeKutta", "[", "]"}]}], ";"}]], "Input",
 CellLabel->"In[50]:=",
 CellID->1833059948],

Cell["\<\
The actual method implementation is written using a stepping procedure.\
\>", "MathCaption",
 CellID->1620292080],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"ClassicalRungeKutta", "[", "___", "]"}], "[", 
    RowBox[{"\"\<Step\>\"", "[", 
     RowBox[{"f_", ",", "t_", ",", "h_", ",", "y_", ",", "yp_"}], "]"}], 
    "]"}], ":=", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"deltay", ",", "k1", ",", "k2", ",", "k3", ",", "k4"}], "}"}], 
     ",", 
     RowBox[{
      RowBox[{"k1", "=", "yp"}], ";", 
      RowBox[{"k2", "=", 
       RowBox[{"f", "[", 
        RowBox[{
         RowBox[{"t", "+", 
          RowBox[{
           RowBox[{"1", "/", "2"}], " ", "h"}]}], ",", 
         RowBox[{"y", "+", 
          RowBox[{
           RowBox[{"1", "/", "2"}], " ", "h", " ", "k1"}]}]}], "]"}]}], ";", 
      RowBox[{"k3", "=", 
       RowBox[{"f", "[", 
        RowBox[{
         RowBox[{"t", "+", 
          RowBox[{
           RowBox[{"1", "/", "2"}], " ", "h"}]}], ",", 
         RowBox[{"y", "+", 
          RowBox[{
           RowBox[{"1", "/", "2"}], " ", "h", " ", "k2"}]}]}], "]"}]}], ";", 
      RowBox[{"k4", "=", 
       RowBox[{"f", "[", 
        RowBox[{
         RowBox[{"t", "+", "h"}], ",", 
         RowBox[{"y", "+", 
          RowBox[{"h", " ", "k3"}]}]}], "]"}]}], ";", 
      RowBox[{"deltay", "=", 
       RowBox[{"h", " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           RowBox[{"1", "/", "6"}], " ", "k1"}], "+", 
          RowBox[{
           RowBox[{"1", "/", "3"}], " ", "k2"}], "+", 
          RowBox[{
           RowBox[{"1", "/", "3"}], " ", "k3"}], "+", 
          RowBox[{
           RowBox[{"1", "/", "6"}], " ", "k4"}]}], ")"}]}]}], ";", 
      RowBox[{"{", 
       RowBox[{"h", ",", "deltay"}], "}"}]}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[51]:=",
 CellID->1239443164],

Cell["\<\
Notice that the implementation closely resembles the description that you \
might find in a textbook. There are no memory allocation/deallocation \
statements or type declarations, for example. In fact the implementation \
works for machine real numbers, machine complex numbers, and even using \
arbitrary precision software arithmetic.\
\>", "Text",
 CellID->1898172925],

Cell["\<\
Here is an example of the calling syntax. For simplicity the method only uses \
fixed step sizes so we need to specify what step sizes to take.\
\>", "MathCaption",
 CellID->2071780786],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{"system", ",", 
   RowBox[{"Method", "\[Rule]", "ClassicalRungeKutta"}], ",", 
   RowBox[{"StartingStepSize", "\[Rule]", 
    RowBox[{"1", "/", "10"}]}]}], "]"}]], "Input",
 CellLabel->"In[52]:=",
 CellID->2102541144],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "20.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{363, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[52]=",
 CellID->105002239]
}, Open  ]],

Cell[TextData[{
 "Many of the methods that have been built into ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " were first prototyped using top-level code before being implemented in the \
kernel for efficiency."
}], "Text",
 CellID->2126723272]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness", "Section",
 CellTags->"s:6",
 CellID->61760426],

Cell["\<\
Stiffness is a combination of problem, initial data, numerical method, and \
error tolerances.\
\>", "Text",
 CellID->2137252429],

Cell["\<\
Stiffness can arise, for example, in the translation of diffusion terms by \
divided differences into a large system of ODEs.\
\>", "Text",
 CellID->488151885],

Cell["\<\
In order to understand more about the nature of stiffness it is useful to \
study how methods behave when applied to a simple problem.\
\>", "Text",
 CellID->1581998586],

Cell[CellGroupData[{

Cell["Linear stability", "Subsection",
 CellID->899442700],

Cell["\<\
Consider applying a Runge-Kutta method to a linear scalar equation known as \
Dahlquist's equation:\
\>", "Text",
 CellID->2096253211],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"y", "'"}], 
     RowBox[{"(", "t", ")"}]}], " ", "=", " ", 
    RowBox[{"\[Lambda]", " ", 
     RowBox[{"y", "(", "t", ")"}]}]}], " ", ",", " ", 
   RowBox[{"\[Lambda]", "\[Element]", "\[DoubleStruckCapitalC]"}], ",", " ", 
   RowBox[{
    RowBox[{"Re", "(", "\[Lambda]", ")"}], " ", "<", " ", "0."}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"DahlquistEquation",
 CellID->1160429658],

Cell[TextData[{
 "The result is a rational function of polynomials ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "z", ")"}], TraditionalForm]]],
 " where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"z", " ", "=", " ", 
    RowBox[{"h", " ", "\[Lambda]"}]}], TraditionalForm]]],
 " (see for example [",
 ButtonBox["L87",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#7030"],
 "])."
}], "Text",
 CellID->1471735303],

Cell[TextData[{
 "This utility function finds the linear stability function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "z", ")"}], TraditionalForm]]],
 " for Runge-Kutta methods. The form depends on the coefficients and is a \
polynomial if the Runge-Kutta method is explicit."
}], "Text",
 CellID->919134292],

Cell["\<\
Here is the stability function for the fifth-order scheme in the \
Dormand-Prince 5(4) pair.\
\>", "MathCaption",
 CellID->985743993],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DOPRIsf", " ", "=", " ", 
  RowBox[{"RungeKuttaLinearStabilityFunction", "[", 
   RowBox[{"DOPRIamat", ",", " ", "DOPRIbvec", ",", "z"}], "]"}]}]], "Input",
 CellLabel->"In[53]:=",
 CellID->373360161],

Cell[BoxData[
 RowBox[{"1", "+", "z", "+", 
  FractionBox[
   SuperscriptBox["z", "2"], "2"], "+", 
  FractionBox[
   SuperscriptBox["z", "3"], "6"], "+", 
  FractionBox[
   SuperscriptBox["z", "4"], "24"], "+", 
  FractionBox[
   SuperscriptBox["z", "5"], "120"], "+", 
  FractionBox[
   SuperscriptBox["z", "6"], "600"]}]], "Output",
 ImageSize->{196, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[53]=",
 CellID->63142193]
}, Open  ]],

Cell[TextData[{
 "This function finds the linear stability function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"R", "(", "z", ")"}], TraditionalForm]]],
 " for Runge-Kutta methods. The form depends on the coefficients and is a \
polynomial if the Runge-Kutta method is explicit."
}], "Text",
 CellID->1550174151],

Cell["\<\
The following package is useful for visualizing linear stability regions for \
numerical methods for differential equations.\
\>", "MathCaption",
 CellID->1266259262],

Cell[BoxData[
 RowBox[{
  RowBox[{"Needs", "[", "\"\<FunctionApproximations`\>\"", "]"}], 
  ";"}]], "Input",
 CellLabel->"In[54]:=",
 CellID->14666232],

Cell[TextData[{
 "We can now visualize the absolute stability region ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"|", 
     RowBox[{"R", "(", "z", ")"}], "|"}], " ", "=", " ", "1"}], 
   TraditionalForm]]],
 "."
}], "MathCaption",
 CellID->515337410],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"OrderStarPlot", "[", 
  RowBox[{"DOPRIsf", ",", " ", "1", ",", "z"}], "]"}]], "Input",
 CellLabel->"In[55]:=",
 CellID->1628247325],

Cell[BoxData[
 GraphicsBox[{GraphicsComplexBox[CompressedData["
1:eJxk3XfYVcX1uH16E1RUsBFrJGoMtgQV+XqeoLH3DtiCiVGCFbtiJdg79l6i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    "], {{
      {GrayLevel[1.], EdgeForm[None], 
       GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxMnXWYl1XTx3+yu6DABkgssUts0AtbsAtIC8LCooSAEoIi2F2omEiYCGKC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          "],
          VertexColors->None], PolygonBox[CompressedData["
1:eJwtnXm8ldP3x480KY3SdJvrNt55qBRnuvece890S0JRGSpDUaQMRcgUJUMi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          "],
          VertexColors->None]}]}, 
      {GrayLevel[0.5], EdgeForm[None], 
       GraphicsGroupBox[{PolygonBox[CompressedData["
1:eJxMnXWYlkUXxt8NtuhYGhZ2F3ZBUunuXMDuxEJBRDpUVFTAFhtEUJRSwMTu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          "],
          VertexColors->None], PolygonBox[CompressedData["
1:eJwtnXdgVEXXxld6AJHeQUgINSQQCEVI2SRbstkNKIJSFBHFQsdCUYqKdAVs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          "],
          VertexColors->None]}]}}, {{}, 
      {GrayLevel[0], Opacity[0.4], 
       TagBox[
        TooltipBox[LineBox[CompressedData["
1:eJwl2nccl+P3BvAPGS2Vym5rF6VtRXunYVRkfolsKltWEcoOkZRSZmZW266k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          "]],
         "1"],
        Annotation[#, 1, "Tooltip"]& ], 
       TagBox[
        TooltipBox[LineBox[CompressedData["
1:eJwl13uYz2Uax/GviBL2isxgtBtyiGJ12O0wmI7O7SbVXirnURLD0BinLcrG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          "]],
         "1"],
        Annotation[#, 1, "Tooltip"]& ], 
       TagBox[
        TooltipBox[LineBox[CompressedData["
1:eJwl2HeUVdUVx/ELDEhJAkiXqihFFJCoKEgbetFZAoMo0ocyNAWTCKNIR/oA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          "]],
         "1"],
        Annotation[#, 1, 
         "Tooltip"]& ]}, {}, {}}}], {{{}, {}}, {{}, {}}, {{}, {}}, {{}, {}}, {
     InsetBox[
      GraphicsBox[
       {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.05]}], 
        PolygonBox[{{-0.7, -0.7}, {0.7, -0.7}, {0.7, 0.7}, {-0.7, 0.7}}]},
       PlotRange->{{-1.05, 1.05}, {-1.05, 1.05}}], {-2.209203507952292, -0.829825069462936},
       Automatic, Scaled[{0.05, 0.05}]], InsetBox[
      GraphicsBox[
       {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.05]}], 
        PolygonBox[{{-0.7, -0.7}, {0.7, -0.7}, {0.7, 0.7}, {-0.7, 0.7}}]},
       PlotRange->{{-1.05, 1.05}, {-1.05, 1.05}}], {-2.209203507952292, 0.829825069462936},
       Automatic, Scaled[{0.05, 0.05}]], InsetBox[
      GraphicsBox[
       {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.05]}], 
        PolygonBox[{{-0.7, -0.7}, {0.7, -0.7}, {0.7, 0.7}, {-0.7, 0.7}}]},
       PlotRange->{{-1.05, 1.05}, {-1.05, 1.05}}], {-1.2784212356187088`, -2.411488029419971},
       Automatic, Scaled[{0.05, 0.05}]], InsetBox[
      GraphicsBox[
       {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.05]}], 
        PolygonBox[{{-0.7, -0.7}, {0.7, -0.7}, {0.7, 0.7}, {-0.7, 0.7}}]},
       PlotRange->{{-1.05, 1.05}, {-1.05, 1.05}}], {-1.2784212356187088`, 2.411488029419971},
       Automatic, Scaled[{0.05, 0.05}]], InsetBox[
      GraphicsBox[
       {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.05]}], 
        PolygonBox[{{-0.7, -0.7}, {0.7, -0.7}, {0.7, 0.7}, {-0.7, 0.7}}]},
       PlotRange->{{-1.05, 1.05}, {-1.05, 1.05}}], {0.9876247435710006, -3.6723915188060055`},
       Automatic, Scaled[{0.05, 0.05}]], InsetBox[
      GraphicsBox[
       {GrayLevel[1], EdgeForm[{GrayLevel[0], Thickness[0.05]}], 
        PolygonBox[{{-0.7, -0.7}, {0.7, -0.7}, {0.7, 0.7}, {-0.7, 0.7}}]},
       PlotRange->{{-1.05, 1.05}, {-1.05, 1.05}}], {0.9876247435710006, 3.6723915188060055`},
       Automatic, Scaled[{0.05, 0.05}]]}}},
  AspectRatio->Automatic,
  Axes->True,
  AxesOrigin->{0, 0},
  DisplayFunction:>$DisplayFunction,
  Frame->True,
  FrameTicks->None,
  ImageMargins->0.,
  ImageSize->Automatic,
  Method->{"AxesInFront" -> True},
  PlotRange->{{-3.6887277456807857`, 2.467148981299495}, {-5.774108974447807, 
   5.774108974447807}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]},
  Ticks->None]], "Output",
 ImageSize->{235, 432},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[55]=",
 CellID->56285197]
}, Open  ]],

Cell[TextData[{
 "Depending on the magnitude of \[Lambda] in (",
 
 CounterBox["NumberedEquation", "DahlquistEquation"],
 "), if we choose the step size ",
 Cell[BoxData[
  FormBox["h", TraditionalForm]]],
 " such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{"R", "(", 
     RowBox[{"h", " ", "\[Lambda]"}], ")"}], "|", " ", 
    RowBox[{"<", " ", "1"}]}], TraditionalForm]]],
 ", then errors in successive steps will be damped and the method is said to \
be absolutely stable."
}], "Text",
 CellID->1662369364],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{"R", "(", 
     RowBox[{"h", " ", "\[Lambda]"}], ")"}], "|", " ", 
    RowBox[{">", " ", "1"}]}], TraditionalForm]]],
 ", then step size selection will be restricted by stability and not by local \
accuracy."
}], "Text",
 CellID->1201927167]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness detection", "Section",
 CellID->1525347659],

Cell[TextData[{
 "The device for stiffness detection that is used with the option ",
 StyleBox["StiffnessTest", "MR"],
 " is described within \"",
 ButtonBox["StiffnessTest Method Option for NDSolve",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveStiffnessTest"],
 "\"."
}], "Text",
 CellID->9198728],

Cell["\<\
Recast in terms of explicit Runge-Kutta methods the condition for stiffness \
detection can be formulated as:\
\>", "Text",
 CellID->560109410],

Cell[BoxData[
 FormBox[
  RowBox[{
   OverscriptBox["\[Lambda]", "\[Tilde]"], " ", "=", " ", 
   FractionBox[
    RowBox[{"\[LeftDoubleBracketingBar]", 
     RowBox[{
      SubscriptBox["k", "s"], "-", 
      SubscriptBox["k", 
       RowBox[{"s", "-", "1"}]]}], "\[RightDoubleBracketingBar]"}], 
    RowBox[{"\[LeftDoubleBracketingBar]", 
     RowBox[{
      SubscriptBox["g", "s"], "-", " ", 
      SubscriptBox["g", 
       RowBox[{"s", "-", "1"}]]}], "\[RightDoubleBracketingBar]"}]]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"StiffnessDevice",
 CellID->1210923959],

Cell[TextData[{
 "with ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["g", "i"], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "i"], TraditionalForm]]],
 " defined in (",
 
 CounterBox["NumberedEquation", "ERKMethod"],
 ")."
}], "Text",
 CellID->1629490808],

Cell[TextData[{
 "The difference ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["g", "s"], TraditionalForm]]],
 " - ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["g", 
    RowBox[{"s", "-", "1"}]], TraditionalForm]]],
 " can be shown to correspond to a number of applications of the power method \
applied to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"h", " ", "J"}], TraditionalForm]]],
 ". "
}], "Text",
 CellID->1799552574],

Cell["\<\
The difference is therefore a good approximation of the eigenvector \
corresponding to the leading eigenvalue.\
\>", "Text",
 CellID->961613236],

Cell[TextData[{
 "The product ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{"h", " ", 
     OverscriptBox["\[Lambda]", "\[Tilde]"]}], "|"}], TraditionalForm]]],
 " gives an estimate that can be compared to the stability boundary in order \
to detect stiffness."
}], "Text",
 CellID->1553787946],

Cell[TextData[{
 "An ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]]],
 "-stage explicit Runge-Kutta has a form suitable for (",
 
 CounterBox["NumberedEquation", "StiffnessDevice"],
 ") if we require that",
 " ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["c", 
     RowBox[{"s", "-", "1"}]], " ", "=", " ", 
    RowBox[{
     SubscriptBox["c", "s"], "=", " ", "1"}]}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1754610070],

Cell[BoxData[
 FormBox[
  TagBox[GridBox[{
     {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
     {
      SubscriptBox["c", "2"], 
      SubscriptBox["a", 
       RowBox[{"2", ",", "1"}]], "0", "\[CenterEllipsis]", "0", "0"},
     {"\[VerticalEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]", 
      "\[DescendingEllipsis]", "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
     {"1", 
      SubscriptBox["a", 
       RowBox[{
        RowBox[{"s", "-", "1"}], ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{
        RowBox[{"s", "-", "1"}], ",", "2"}]], "\[CenterEllipsis]", "0", "0"},
     {"1", 
      SubscriptBox["a", 
       RowBox[{"s", ",", "1"}]], 
      SubscriptBox["a", 
       RowBox[{"s", ",", "2"}]], "\[CenterEllipsis]", 
      SubscriptBox["a", 
       RowBox[{"s", ",", 
        RowBox[{"s", "-", "1"}]}]], "0"},
     {"", 
      SubscriptBox["b", "1"], 
      SubscriptBox["b", "2"], "\[CenterEllipsis]", 
      SubscriptBox["b", 
       RowBox[{"s", "-", "1"}]], 
      SubscriptBox["b", "s"]}
    },
    GridBoxDividers->{
     "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, False, False, False, False, {True}, False}, 
      "RowsIndexed" -> {}}],
   DisplayForm], TraditionalForm]], "NumberedEquation",
 CellTags->"StiffButcherTable",
 CellID->1723376525],

Cell[TextData[{
 "The default embedded pairs used in ",
 StyleBox["ExplicitRungeKutta", "MR"],
 " all have the form ",
 "(",
 
 CounterBox["NumberedEquation", "StiffButcherTable"],
 ")",
 "."
}], "Text",
 CellID->1533027570],

Cell[TextData[{
 "An important point is that (",
 
 CounterBox["NumberedEquation", "StiffnessDevice"],
 ") is very cheap and convenient; it uses already-available information from \
the integration and requires no additional function evaluations."
}], "Text",
 CellID->1816370496],

Cell[TextData[{
 "Another advantage of (",
 
 CounterBox["NumberedEquation", "StiffButcherTable"],
 ") is that it is straightforward to make use of consistent FSAL methods (",
 
 CounterBox["NumberedEquation", "FSALConditions"],
 ")."
}], "Text",
 CellID->1713145306],

Cell[CellGroupData[{

Cell["Examples", "Subsection",
 CellID->2021179456],

Cell["Select a stiff system modeling a chemical reaction.", "MathCaption",
 CellID->1250885435],

Cell[BoxData[
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<Robertson\>\"", "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[56]:=",
 CellID->465104358],

Cell["\<\
This applies a built-in explicit Runge-Kutta method to the stiff system.\
\>", "MathCaption",
 CellID->1969949019],

Cell["\<\
By default stiffness detection is enabled, since it only has a small impact \
on the running time.\
\>", "MathCaption",
 CellID->263831445],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{"system", ",", " ", 
   RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[57]:=",
 CellID->1936839661],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ndstf\"\>"}], ":", 
  " ", "\<\"At \\!\\(T\\) == \\!\\(0.0365361739659134`\\), system appears to \
be stiff. Methods Automatic, BDF or StiffnessSwitching may be more \
appropriate.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[57]:=",
 CellID->185903983],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.0365361739659134`"}], "}"}], "}"}], 
         ",", "\<\"<>\"\>"}], "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.0365361739659134`"}], "}"}], "}"}], 
         ",", "\<\"<>\"\>"}], "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "3"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.0365361739659134`"}], "}"}], "}"}], 
         ",", "\<\"<>\"\>"}], "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{405, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[57]=",
 CellID->277105984]
}, Open  ]],

Cell[TextData[{
 "The coefficients of the Dormand-Prince 5(4) pair are of the form (",
 
 CounterBox["NumberedEquation", "StiffButcherTable"],
 "). However, using these coefficients gives the following message. "
}], "MathCaption",
 CellID->1249575439],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "[", 
   RowBox[{"system", ",", " ", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
       RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "5"}], ",", 
       RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "DOPRICoefficients"}], ",", 
       " ", 
       RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "True"}]}], "}"}]}]}], 
   "]"}], ";"}]], "Input",
 CellLabel->"In[58]:=",
 CellID->162716174],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"mtdp\"\>"}], ":", 
  " ", "\<\"\\!\\(DOPRICoefficients\\) does not have a correctly defined \
property \\!\\(\\\"LinearStabilityBoundary\\\"\\) in \
\\!\\(\\\"ExplicitRungeKutta\\\"\\).\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[58]:=",
 CellID->222257229],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"initf\"\>"}], ":", 
  " ", "\<\"The initialization of the method \\!\\(NDSolve`ExplicitRungeKutta\
\\) failed.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[58]:=",
 CellID->515687557]
}, Open  ]],

Cell["\<\
The reason is that the stiffness detection device needs to know where the \
border of the linear stability boundary is. Therefore, we need to determine \
the point at which the stability region crosses the negative real axis.\
\>", "Text",
 CellID->737236628],

Cell["\<\
We can set up an equation in terms of the linear stability function and solve \
it exactly to find the point where the contour crosses the negative real \
axis.\
\>", "MathCaption",
 CellID->1428975437],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Abs", "[", "DOPRIsf", "]"}], " ", "\[Equal]", " ", "1"}], " ", "&&",
     " ", 
    RowBox[{"z", " ", "<", " ", "0"}]}], ",", "z"}], "]"}]], "Input",
 CellLabel->"In[59]:=",
 CellID->751902860],

Cell[BoxData[
 RowBox[{"z", "\[Equal]", 
  RowBox[{"Root", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"600", "+", 
      RowBox[{"300", " ", "#1"}], "+", 
      RowBox[{"100", " ", 
       SuperscriptBox["#1", "2"]}], "+", 
      RowBox[{"25", " ", 
       SuperscriptBox["#1", "3"]}], "+", 
      RowBox[{"5", " ", 
       SuperscriptBox["#1", "4"]}], "+", 
      SuperscriptBox["#1", "5"]}], "&"}], ",", "1"}], "]"}]}]], "Output",
 ImageSize->{350, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[59]=",
 CellID->98308044]
}, Open  ]],

Cell["\<\
In general, there may be more than one point of intersection and it may be \
necessary to choose the appropriate solution.\
\>", "Text",
 CellID->524141184],

Cell["\<\
The following definition sets the value of the linear stability boundary.\
\>", "MathCaption",
 CellID->1725897629],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"DOPRICoefficients", "[", "5", "]"}], "[", 
    "\"\<LinearStabilityBoundary\>\"", "]"}], "=", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"600", "+", 
       RowBox[{"300", "*", "#1"}], "+", 
       RowBox[{"100", "*", 
        RowBox[{"#1", "^", "2"}]}], "+", 
       RowBox[{"25", "*", 
        RowBox[{"#1", "^", "3"}]}], "+", 
       RowBox[{"5", "*", 
        RowBox[{"#1", "^", "4"}]}], "+", 
       RowBox[{"#1", "^", "5"}]}], "&"}], ",", "1", ",", "0"}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[60]:=",
 CellID->682489069],

Cell[TextData[{
 "The coefficients can now be used with stiffness detection enabled for the \
default setting ",
 StyleBox["StiffnessTest\[Rule]Automatic", "MR"],
 "."
}], "MathCaption",
 CellID->1726068352],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NDSolve", "[", 
  RowBox[{"system", ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
      RowBox[{"DifferenceOrder", "\[Rule]", "5"}], ",", 
      RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "DOPRICoefficients"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[61]:=",
 CellID->1103761640],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ndstf\"\>"}], ":", 
  " ", "\<\"At \\!\\(T\\) == \\!\\(0.03632977478700775`\\), system appears to \
be stiff. Methods Automatic, BDF or StiffnessSwitching may be more \
appropriate.\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[61]:=",
 CellID->57312333],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.03632977478700775`"}], "}"}], "}"}], 
         ",", "\<\"<>\"\>"}], "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.03632977478700775`"}], "}"}], "}"}], 
         ",", "\<\"<>\"\>"}], "]"}],
       False,
       Editable->False], "[", "T", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "3"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"0.`", ",", "0.03632977478700775`"}], "}"}], "}"}], 
         ",", "\<\"<>\"\>"}], "]"}],
       False,
       Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{405, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[61]=",
 CellID->952136848]
}, Open  ]],

Cell[TextData[{
 "The Fehlberg 4(5) method does not have the correct coefficient structure (",
 
 
 CounterBox["NumberedEquation", "StiffButcherTable"],
 ") required for stiffness detection, since ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["c", "s"], " ", "=", 
    RowBox[{
     RowBox[{"1", "/", "2"}], "\[NotEqual]", " ", "1"}]}], 
   TraditionalForm]]],
 "."
}], "Text",
 CellID->167201951],

Cell[TextData[{
 "The default value ",
 StyleBox["StiffnessTest\[Rule]Automatic", "MR"],
 " checks to see if the method coefficients provide a stiffness detection \
capability and if they do then stiffness detection is enabled."
}], "Text",
 CellID->173694957]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Step control revisited", "Section",
 CellTags->"s:7",
 CellID->1938246688],

Cell[TextData[{
 "There are some reasons to look at alternatives to the standard Integral \
step controller (",
 
 CounterBox["NumberedEquation", "IController"],
 ") when considering mildly stiff problems."
}], "Text",
 CellID->1128815187],

Cell["This system models a chemical reaction.", "MathCaption",
 CellID->1727482903],

Cell[BoxData[
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<Robertson\>\"", "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[62]:=",
 CellID->1545373110],

Cell["\<\
This defines an explicit Runge-Kutta method based on the Dormand-Prince \
coefficients that does not use stiffness detection.\
\>", "MathCaption",
 CellID->704708064],

Cell[BoxData[
 RowBox[{
  RowBox[{"IERK", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
     RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "DOPRICoefficients"}], ",", 
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "5"}], ",", 
     RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}]}], "}"}]}], 
  ";"}]], "Input",
 CellLabel->"In[63]:=",
 CellID->1113026826],

Cell[TextData[{
 "This solves the system and plots the step sizes that are taken using the \
utility function ",
 StyleBox["StepDataPlot", "MR"],
 "."
}], "MathCaption",
 CellID->1214259959],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"isol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", " ", 
     RowBox[{"Method", "\[Rule]", "IERK"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"StepDataPlot", "[", "isol", "]"}]}], "Input",
 CellLabel->"In[64]:=",
 CellID->270369723],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[65]=",
 CellID->175811298]
}, Open  ]],

Cell["\<\
Solving a stiff or mildly stiff problem with the standard step size \
controller leads to large oscillations, sometimes leading to a number of \
undesirable step size rejections.\
\>", "Text",
 CellID->317661746],

Cell["The study of this issue is known as step-control stability.", "Text",
 CellID->735474081],

Cell["\<\
It can be studied by matching the linear stability regions for the high- and \
low-sorder methods in an embedded pair.\
\>", "Text",
 CellID->809188005],

Cell["\<\
One approach to addressing the oscillation is to derive special methods, but \
this compromises the local accuracy.\
\>", "Text",
 CellID->581493191],

Cell[CellGroupData[{

Cell["PI step control", "Subsection",
 CellID->524830095],

Cell[TextData[{
 "An appealing alternative to Integral step control (",
 
 CounterBox["NumberedEquation", "IController"],
 ") is Proportional-Integral or PI step control."
}], "Text",
 CellID->1550787669],

Cell["\<\
In this case the step size is selected using the local error in two \
successive integration steps according to the formula:\
\>", "Text",
 CellID->1831068630],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["h", 
    RowBox[{"n", "+", "1"}]], "=", " ", 
   RowBox[{
    SubscriptBox["h", "n"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox["Tol", 
       RowBox[{"\[LeftDoubleBracketingBar]", 
        SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}]], ")"}], 
     RowBox[{
      SubscriptBox["k", "1"], "/", " ", 
      OverscriptBox["p", "\[Tilde]"]}]], 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox[
       RowBox[{"\[LeftDoubleBracketingBar]", 
        SubscriptBox["err", 
         RowBox[{"n", "-", "1"}]], "\[RightDoubleBracketingBar]"}], 
       RowBox[{"\[LeftDoubleBracketingBar]", 
        SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}]], ")"}], 
     RowBox[{
      SubscriptBox["k", "2"], " ", "/", " ", 
      OverscriptBox["p", "\[Tilde]"]}]]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"PIController",
 CellID->687546269],

Cell["\<\
This has the effect of damping and hence gives a smoother step size sequence.\
\
\>", "Text",
 CellID->140540650],

Cell[TextData[{
 "Note that Integral step control (",
 
 CounterBox["NumberedEquation", "IController"],
 ") is a special case of (",
 
 CounterBox["NumberedEquation", "PIController"],
 ") and is used if a step is rejected:"
}], "Text",
 CellID->335850925],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["k", "1"], "=", "1"}], ",", 
   RowBox[{
    SubscriptBox["k", "2"], "=", 
    RowBox[{"0", " ", "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->2116521706],

Cell[TextData[{
 "The option ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     RowBox[{"StepSizeControlParameters", "\[Rule]", 
      RowBox[{"{", 
       SubscriptBox["k", "1"]}]}],
     TraditionalForm], ",", 
    SubscriptBox["k", "2"]}], TraditionalForm]]],
 StyleBox["}", "MR"],
 " can be used to specify the values of ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "1"], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["k", "2"], TraditionalForm]]],
 "."
}], "Text",
 CellID->892443510],

Cell[TextData[{
 "The scaled error estimate in (",
 
 CounterBox["NumberedEquation", "PIController"],
 ") is taken to be ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FormBox[
     RowBox[{"\[LeftDoubleBracketingBar]", 
      SubscriptBox["err", 
       RowBox[{"n", "-", "1"}]], "\[RightDoubleBracketingBar]"}],
     TraditionalForm], " ", "=", " ", 
    FormBox[
     RowBox[{"\[LeftDoubleBracketingBar]", 
      SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}],
     TraditionalForm]}], TraditionalForm]]],
 " for the first integration step."
}], "Text",
 CellID->1503454278]
}, Open  ]],

Cell["Examples", "Subsection",
 CellID->493179243],

Cell[CellGroupData[{

Cell["Stiff problem", "Subsection",
 CellID->470857491],

Cell[TextData[{
 "This defines a method similar to ",
 StyleBox["IERK", "MR"],
 " that uses the option ",
 StyleBox["StepSizeControlParameters", "MR"],
 " to specify a PI controller."
}], "Text",
 CellID->38459699],

Cell["\<\
Here we use generic control parameters suggested by Gustafsson:\
\>", "Text",
 CellID->71763947],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      RowBox[{
       SubscriptBox["k", "1"], "=", " ", 
       RowBox[{"3", "/", "10"}]}], ","}], 
     RowBox[{
      SubscriptBox["k", "2"], "=", " ", 
      RowBox[{
       RowBox[{"2", "/", "5"}], 
       FormBox["",
        TraditionalForm]}]}]}
   }], TraditionalForm]], "DisplayMath",
 CellID->1574619131],

Cell["This specifies the step control parameters.", "MathCaption",
 CellID->730147496],

Cell[BoxData[
 RowBox[{
  RowBox[{"PIERK", " ", "=", 
   RowBox[{"{", 
    RowBox[{"\"\<ExplicitRungeKutta\>\"", ",", 
     RowBox[{"\"\<Coefficients\>\"", "\[Rule]", "DOPRICoefficients"}], ",", 
     RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "5"}], ",", 
     RowBox[{"\"\<StiffnessTest\>\"", "\[Rule]", "False"}], ",", 
     RowBox[{"\"\<StepSizeControlParameters\>\"", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"3", "/", "10"}], ",", " ", 
        RowBox[{"2", "/", "5"}]}], "}"}]}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[66]:=",
 CellID->210458061],

Cell["\<\
Solving the system again, it can be observed that the step size sequence is \
now much smoother.\
\>", "MathCaption",
 CellID->1741821082],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pisol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", " ", 
     RowBox[{"Method", "\[Rule]", "PIERK"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"StepDataPlot", "[", "pisol", "]"}]}], "Input",
 CellLabel->"In[67]:=",
 CellID->903842453],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 104},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[68]=",
 CellID->93585090]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Nonstiff problem", "Subsection",
 CellID->723152621],

Cell[TextData[{
 "In general the I step controller (",
 
 CounterBox["NumberedEquation", "IController"],
 ") is able to take larger steps for a nonstiff problem than the PI step \
controller (",
 
 CounterBox["NumberedEquation", "PIController"],
 ") as the following example illustrates."
}], "Text",
 CellID->483841708],

Cell["\<\
Select and solve a non-stiff system using the I step controller.\
\>", "MathCaption",
 CellID->1546594211],

Cell[BoxData[
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<BrusselatorODE\>\"", "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[69]:=",
 CellID->1427860686],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"isol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", " ", 
     RowBox[{"Method", "\[Rule]", "IERK"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"StepDataPlot", "[", "isol", "]"}]}], "Input",
 CellLabel->"In[70]:=",
 CellID->1596868534],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW2uMVdUVPsx9DQwgSn20KDAMItXG1NL2nwzzgBnAAcQRFBkVCMpTELEq
+KhakOCjxmpk5g4Mw0NAqDCAYpEq4qPa+mgb6cumrU0a+6OmSU1/mDS5XXt/
66y1z77nXO5t0aRJJ3Dv2Wc/1rfWXs99zm2bt3rRwuXzVi9eMG9ky6p5Kxct
XnDbyEkrVtGtVL8gCA7T/5XnBXRdoCv7/3ChQN/u5zn0WUVddfSdou9a/q7i
+8O5HaTtcP4zd9POqJTtPNt8mtsF+0cXF5qvlN4ZUTSiyhtxvtdOxfVn0JZP
u4phuZ/5/sKwXBC7whePY7jTLlcadbxzo8qjriN4Xu1/SNNdubY82hmXtqFl
JZZ15yVJP9TuC/l7ZEH1NdTuKlZut50quHpdbDXup+Gzu4QN5Dye0l478z/e
/3nSCyJyrdKm7fymnd7cmK+/vJP+NTduPtNby1pGesL4Lox49JHXxhTgz5zd
csZ/mz7SNPAyC+T21S9gnv3XFYz1RveLXeNbti1zgyrzkbqhYy/3X2r7BZPt
r2RlnZmij6/j3uXlrWYkiFVEaoHh91IPc9rD7PuseHzjzEfWldr1c/cQ3KqQ
Qkaw23+bLAcXVbC2O39iU/fUyVshXdpYHneJ+agWedSP79q44XhTA5jtqoBW
/3CNTfS58pZDb7/154Z6c6ehPh98zYzI7dzxvouG7tGIwEc0uiya37BtWRGr
ZLBqy8TN1JpAa1u6A66YshXsdMzdc+SF39C9bb3vQjQEbkzJ3R/nrjC+86kn
31xxyyEs0jppiyEzvstSSU22zYbxnWyhk5pDFKE3jbchcDJj2jYa3Tihi2fL
4uAhBZ50de6n1ZPxF1PZ8NBxmrVi+UGskgZVWPrF9o5ItKmhK+0hK4+SjUnp
uXN2y8osn5YeK3DmME0K0tTQDVUA7dlXPwOeSb5pB+8NHc+WoF0VxsGI/xGt
S01t7QE77Ckf+O4xaGhwsUNjwfx93A8Uht+YfePYCimRxluLDFJkMIzrq44U
J7f2ECfuqsQqU9V2JpTGldO3Q17x3iP05OLUXZmzDMwtu15/9LFErUQmjDef
t9DeX+IgIORB1pHDzQufY4T3i5zQP3PGDkacgC8SxbHHRXd8PWGcxD3MpWXS
lmlX9NrglGecwLVsyQHGobiBE2IzuFw+DJ+nEedAaTdOyEONCfPUKVs7n34L
akfOH3Hk6qt2unjMzoIl2tmxjna0TNxC2uHyZHg+fZhtFZJttq6ctJ/ukOd6
ZOOrKc/fguL+H37Q2mIAcf4FqzdcjI1YRZ771TJHR6zkVEjDCshqabX03Hbr
8xObNsOvzp+3D/GUbJjnISZAbnOu2R1U2/Z9977EGgqMRtJtvU88/jpjjFpy
5VL1s2s7QrPrh9YfJ4myv4M/hvvyI1qdh8aiz6AtTpI9HHgk22MeiKfp03o1
L9JVR8bz5HiJbCGarRfjN3Qs/lzHdXs4K2zIhyoeaqPxTTO2U17Acqd2JFeL
59ToDjiVKMOrjopFLhlrBjJANkXXTIViHekHrZVAlfbi/ff+YplxJKkoVZsw
/thLHzY1dhuNB1dr73wxgSu3yuE4cIbrX+H93ehVsC6NKDLHrv4++MDLoJgV
jWhiI81D3ic/+OviRQesgTjjaGMkoXKryagEHY8qiYpcc9zQ3GJ03H7lZDxd
YKzRzzrPBvtbbDJ2ets28FACm3PXzmYvY3KC0d6eDLRtLGqQ+TaEfnih+Tfu
C5Kpmp0bUPBjbJ0naay3ft0rHDvqQk1vv2rng/cf437QN3jL49LSdSICVn3t
xB/XrvmRUtWIBilQJw0xlzUh7lntO4E73upj6aYlqtK8Wmf35pG8BoUjrpi6
1ZUIiYOE4mNrbuxO8Dc+ZYs4RYUGc1wb2bd8MNi2NWZnHFxGy4aF4z/++B+C
wuCc3MKKmPZmnFcBrl0736eYbfSrNtxfjhiDvf1POVTqY6rScz0UmI9Ab/jC
fL+qVZ/ypZL6Ctn3bHmHV4fndusnc/fLdhRlrh/+7m/mclBkx/KMIax+OgM/
8p9j+9tnhnkdeLhn7VGOqpjfWJ+vj/ejQwul6poaD03a0Yw4iQAN4fjss38p
N2qBNk5lRQaUfE9v642JtsCxd88vkH4zV+rjqhxZNkzoapuylTgsQjO0Aq0i
jDOv3LF+3cuQhFvndkmZCf60viJSZ1g2133vZTQLEkPq3RgSzofG/fjY7ylo
m0vLV8ZacC+VpQUnVqZRB1MfXficnVUWZ1YDM/fdc1SwlROJrdTSktkQdctj
TnQM7Dc25CUaRWrQc0Kr6tz0trnE7FW3Hka3G2I57yIeJV0pbVNWn2qkiomV
ttWPOHkZOs0N3ddes4vGWlRZUD6w/wMJSk5dmwcR8ENr9R04KfxkJaoZBz1x
847t7xVX42eVtC27OyZOE/hDB39FreOv/IHQqFRFLkTmbLa8nk8//UxRYCvu
+M4R41xbezbnfyqVaHNTN1AMjeosDR5i5wq1TU+/5XhlO7eRa/p4CypVxbrP
UfxTz4EO5rvXHMXF7FnPYBz89tIlfSwpavA88CpbJC4VdVfb1NBfe5pLXNiz
0Qwikfp1kZFE1nh7ilYSqARL1RZ+e2AEd14uomcU+QR+CR+FhY65e0TVso6c
THwbGvJB4YEG0VYOcSS86tbnqSLDcRlmaib2efJLeK7veHbRzc+ZzLNlS2/P
OyJvyYCgV7QpJARziX3a1vsudc9qf2bOtbuNFfhnBfANVPLA1oMh4e4in+O8
POdIafnSvtPIayr0QBmYzKKb9tPnwxtPFCGN5U818957XsqFMqGqzJUJZRt0
U7kTqfHO5TxKyAHdiJRUx1e+n4Mc+SoCORFBTSvWo3n4meEI8SPVDmqxVKfa
c+WD0FAsH9eaxP2fZl5Bb0przzWzd+k5iub+4JRAE/RCGFdSWptVh6gl08UI
zcjPDGcsW9pnZqCGIp1nCmJDlBjBpiNcxpwMhM/tfG+LnOmB+4/VO/VqxOsJ
FlI6c4lnqr6X0Dz+gnD89GnbHnvkBMtM7JGhRZ9dneHxB36Rz5o2MNACtIxi
oNhENSRFxSIM6JechrNs9QnJ8Xawt1dAcujgryl8m7aPZIRtXzt7F1NCv3pp
INHMG9kWJXaU3sVJw8XyFdvWswSzdhXFtZGx0vFpimNlmmJHZdBUzxGlqWeH
1u+kKM2gZMNcon/F8oMsh5pT0DQ7MCzEeeMNexcu2Md8/OCJN/c++0td08tm
HH+Jfq0wazwrSt7lYR76IZ5+jHAkf/ttL3C/T/mN1/+ElJK5Ff0rKeFhztrk
4nlt0Qimrecf6FdvgnpWa8EBFdPW3cXa6mFHhDty08LnZrXvSKC9bEkf7zJo
0w588sk/y6NNa1OyQW4NlqYx6a47X8RpodaFsEU9QwF19TygLnofq2PwRh99
9PeO6/aYS0RCNz645yUEgrVMLRz9Lx75LY6t+aRHrS1Zy853Vt7w0Kt6EjOp
uZsSWlosembSVUR7lGcBiJAkHo5rfnyKr/DO93Zd6mT2lnJCwnZoUBibH+VZ
QX/PCiqhrtW9nAtBZ5hHXXVwhLrKpnzq4fsqbg09yNu1Ok+yNVFtXL7s4ChP
KpZ+WjSpUKpqg9aRgZKZxlF3JUulFp99iVbSJtV53GLn1RLjuT9VzRV9V83P
BYAa+bXJVwd6GGJPZtM4Or66fSep7WhPqrnyUBdnZS5ur+2vcYGDypw+xaOW
eoP3Qv0nMPvPFDTf+fwwa44EzJobj/b0R05o2WLRr34YmNVvnC7MKbGkzk1v
k7oWQktx9hRYNJYO9HQA/Vp/I2cXrIXkZ4pefl6Iz2B9Swc+jRp40qL+Bv1q
i8Cnp4iQZd+BkzgYKl+WRW/j+RIe7u1yjbdr7rMj4xVqPMmCE2KDmCmwphpP
6e9cmc8Hi5AppRpPBu7TI3MWgX61sTHenmdLIgt9tJ/fDeBZ8iySKvzVqw4z
NbUGUFMLhp6IhhZKeebhjozvXnPUrm1ojvG0BlFetRz9mqVhb52KIpHTYuoa
iWu8VS70OB3goQIKtcX/BgXJl5w21UgJKFQbBnh6ChRaXWU92ZSOzNh1SvEp
0TeX/nMhzNYTDl+3rO7m3LMMuoHlArzFFv/MyeR2MW9lOzhHeHy6T4aoHOYT
dxBEvLjI00Q8vUEdWx8+Lg6MfmGk1pE4RRUumhu7k22lGKPqpZ3VP3z7pwtn
KaALWCwTfdII+T368AkqIEmqECGKLh6rHgYowQ9KnUpQatYNlPIeDV1cN2f3
SssBghpT1trPzqjGIe76da+IyUBL8RaI+nCgeurJN6+csV2kWyZK5RYoxfro
TzwcwgejvOuOI6yrQCmVuCQZkLa8T8NRGyi1Pk1GKe+R59zqBtqydHEfPxER
L1YfvtmqtRGQzb9xL/ECswZf0Nuxnt7C5p54/I16fQqZbCfWgnNuxek9c0tL
jijI9HksaKl3wQyqaahN6kKIMEM9AGbok2Y/A4jfXaAUHulP3zmQJ0o337S/
ZWI3wR3raSxoalaRCiVGGIy+YrzveZcsPkDVgqFQAcascEp/8PVmdlWIKica
KjpgntlCe/DuHkmTEWlsqPK0LxmRaFvWPaXDsznM0kzTUsmKVspLA+6zenOJ
ceLl6Q877p5HJslDTpGohagCFPpeIlaXnE4OYtx3SRWtm2sir/d/eVC6erLt
IPmJFVCLF6AWKRGpEsbpWaJFk0FfITRIXk1Pj4FZPHwhDG4JmCN5qF0r9jdA
paoRiz8jForjhQBv+ktdBFMIYnnKPvXkT7wdKwdt5efvkLT4ju8/+hrI2nf9
9VQVqhvYu1mRrBy5lYPNHZMkw1RoN7yneiILPPI0GMAYj334vHn3rp8LsKhN
VI7Dbdc6khDvbymbLMSS4LwA+wuL39L9s2ltvVr9lJaNtbKEXxmdEpvJ/Age
2S7eVM9IlYGDP+BK4y7QVcdjquApByiTIhMN4xcs5TRUGt4Vd2CS8JnxVEN/
iRMc5CcmswQvsFr1P3gXn5SV66RcoZT3c08DjRZhTeQgeDtP7DIlb07yudip
Io2c8bHdYm3gb5+5g+9e5iAwb4KVt/YsD/c4j1byKv7vlvxfhPjvj1Z7K5/r
9Ye/1fz/L85OP70gIlf9XV/Q798QtzUM\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[71]=",
 CellID->507999994]
}, Open  ]],

Cell["\<\
Using the PI step controller the step sizes are slightly smaller.\
\>", "MathCaption",
 CellID->1698230409],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pisol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", " ", 
     RowBox[{"Method", "\[Rule]", "PIERK"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"StepDataPlot", "[", "pisol", "]"}]}], "Input",
 CellLabel->"In[72]:=",
 CellID->436107520],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 115},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[73]=",
 CellID->618346149]
}, Open  ]],

Cell[TextData[{
 "For this reason, the default setting for ",
 StyleBox["StepSizeControlParameters", "MR"],
 " is ",
 StyleBox["Automatic", "MR"],
 " , which is interpreted as:"
}], "Text",
 CellID->286220842],

Cell[TextData[{
 "Use the I step controller (",
 
 CounterBox["NumberedEquation", "IController"],
 ") if ",
 StyleBox["StiffnessTest \[Rule] False", "MR"],
 "."
}], "BulletedText",
 CellID->1245295526],

Cell[TextData[{
 "Use the PI step controller (",
 
 CounterBox["NumberedEquation", "PIController"],
 ") if ",
 StyleBox["StiffnessTest \[Rule] True", "MR"],
 "."
}], "BulletedText",
 CellID->1512525460]
}, Open  ]],

Cell[CellGroupData[{

Cell["Fine-tuning", "Subsection",
 CellID->811050937],

Cell[TextData[{
 "Instead of using (",
 
 CounterBox["NumberedEquation", "PIController"],
 ") directly, it is common practice to use safety factors to ensure that the \
error is acceptable at the next step with high probability, thereby \
preventing unwanted step rejections."
}], "Text",
 CellID->648599547],

Cell[TextData[{
 "The option ",
 StyleBox["StepSizeSafetyFactors\[Rule]{", "MR"],
 StyleBox[Cell[BoxData[
  FormBox[
   SubscriptBox["s", "1"], TraditionalForm]], "MR"], "MR"],
 StyleBox[",", "MR"],
 StyleBox[Cell[BoxData[
  FormBox[
   SubscriptBox["s", "2"], TraditionalForm]], "MR"], "MR"],
 StyleBox["}", "MR"],
 " specifies the safety factors to use in the step size estimate so that (",
 
 CounterBox["NumberedEquation", "PIController"],
 ")",
 " ",
 "becomes:"
}], "Text",
 CellID->1196110442],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["h", 
    RowBox[{"n", "+", "1"}]], "=", " ", 
   RowBox[{
    SubscriptBox["h", "n"], " ", 
    SubscriptBox["s", "1"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox[
       RowBox[{
        SubscriptBox["s", "2"], "Tol"}], 
       RowBox[{"\[LeftDoubleBracketingBar]", 
        SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}]], ")"}], 
     RowBox[{
      SubscriptBox["k", "1"], "/", " ", 
      OverscriptBox["p", "\[Tilde]"]}]], 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       FractionBox[
        RowBox[{"\[LeftDoubleBracketingBar]", 
         SubscriptBox["err", 
          RowBox[{"n", "-", "1"}]], "\[RightDoubleBracketingBar]"}], 
        RowBox[{"\[LeftDoubleBracketingBar]", 
         SubscriptBox["err", "n"], "\[RightDoubleBracketingBar]"}]], ")"}], 
      RowBox[{
       SubscriptBox["k", "2"], "/", " ", 
       OverscriptBox["p", "\[Tilde]"]}]], " ", "."}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"StepSizeSafetyFactors",
 CellID->1498597207],

Cell[TextData[{
 "Here ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["s", "1"], TraditionalForm]]],
 " is an absolute factor and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["s", "2"], TraditionalForm]]],
 " typically scales with the order of the method."
}], "Text",
 CellID->789140197],

Cell[TextData[{
 "The option ",
 StyleBox["StepSizeRatioBounds\[Rule]{", "MR"],
 StyleBox[Cell[BoxData[
  FormBox[
   SubscriptBox["sr", "min"], TraditionalForm]], "MR"], "MR"],
 StyleBox[",", "MR"],
 StyleBox[Cell[BoxData[
  FormBox[
   SubscriptBox["sr", "max"], TraditionalForm]], "MR"], "MR"],
 StyleBox["}", "MR"],
 " specifies bounds on the next step size to take such that:"
}], "Text",
 CellID->1531961367],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["sr", "min"], "\[LessEqual]", 
   RowBox[{"\[LeftBracketingBar]", 
    FractionBox[
     SubscriptBox["h", 
      RowBox[{"n", "+", "1"}]], 
     SubscriptBox["h", "n"]], "\[RightBracketingBar]"}], "\[LessEqual]", 
   SubscriptBox["sr", "max"]}], TraditionalForm]], "NumberedEquation",
 CellTags->"StepSizeRatioBounds",
 CellID->1467635265]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
 CellTags->"s:8",
 CellID->1681583708],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"Coefficients", "EmbeddedExplicitRungeKuttaCoefficients", Cell["\<\
specifies the coefficients of the explicit Runge-Kutta method\
\>", "TableText"]},
   {"DifferenceOrder", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies the order of local accuracy.", "TableText"]},
   {"EmbeddedDifferenceOrder", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
specifies the order of the embedded method in a pair of explicit Runge-Kutta \
methods\
\>", "TableText"]},
   {"StepSizeControlParameters", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "specifies the PI step control parameters (see (",
     
     CounterBox["NumberedEquation", "PIController"],
     "))"
    }], "TableText"]},
   {"StepSizeRatioBounds", 
    RowBox[{"{", 
     RowBox[{
      FractionBox["1", "8"], ",", "4"}], "}"}], Cell[TextData[{
     "specifies the bounds on a relative change in the new step size (see (",
     
     CounterBox["NumberedEquation", "StepSizeRatioBounds"],
     "))"
    }], "TableText"]},
   {"StepSizeSafetyFactors", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "specifies the safety factors to use in the step size estimate (see (",
     
     CounterBox["NumberedEquation", "StepSizeSafetyFactors"],
     "))"
    }], "TableText"]},
   {"StiffnessTest", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies whether to use the stiffness detection capability", 
     "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->308103003],

Cell[TextData[{
 "Options of the method ",
 Cell[BoxData["ExplicitRungeKutta"], "InlineFormula"],
 ". "
}], "Caption",
 CellID->1960814508],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["DifferenceOrder", "MR"],
 " selects the default coefficient order based on the problem, initial values \
and local error tolerances, balanced against the work of the method for each \
coefficient set."
}], "Text",
 CellID->107132032],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["EmbeddedDifferenceOrder", "MR"],
 " specifies that the default order of the embedded method is one lower than \
the method order. This depends on the value of the ",
 StyleBox["DifferenceOrder", "MR"],
 " option."
}], "Text",
 CellID->501489466],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["StepSizeControlParameters", "MR"],
 " uses the values ",
 StyleBox["{1,0}", "MR"],
 " if stiffness detection is active and ",
 StyleBox["{3/10,2/5}", "MR"],
 " otherwise."
}], "Text",
 CellID->12788916],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["StepSizeSafetyFactors", "MR"],
 " uses the values ",
 StyleBox["{17/20,9/10}", "MR"],
 " if the I step controller (",
 
 CounterBox["NumberedEquation", "IController"],
 ") is used and ",
 StyleBox["{9/10,9/10}", "MR"],
 " if the PI step controller (",
 
 CounterBox["NumberedEquation", "PIController"],
 ") is used. The step controller used depends on the values of the options ",
 StyleBox["StepSizeControlParameters", "MR"],
 " and ",
 StyleBox["StiffnessTest", "MR"],
 "."
}], "Text",
 CellID->623800798],

Cell[TextData[{
 "The default setting of ",
 StyleBox["Automatic", "MR"],
 " for the option ",
 StyleBox["StiffnessTest", "MR"],
 " will activate the stiffness test if a ",
 StyleBox["LinearStabilityBoundary", "MR"],
 " method property is specified and if the coefficients have the form (",
 
 CounterBox["NumberedEquation", "StiffButcherTable"],
 ")."
}], "Text",
 CellID->6412433],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->77323823],

Cell[TextData[{
 ButtonBox["Advanced Numerical Differential Equation Solving in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
  FontSlant->"Italic"]
}], "RelatedTutorials",
 CellID->604721514]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ExplicitRungeKutta Method for NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveDesign#590065589"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveImplicitRungeKutta"]}],
       "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 46.5243980}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "This loads packages containing some test problems and utility functions. \
One of the first and simplest methods for solving initial value problems was \
proposed by Euler: Euler's method is not very accurate.", "synonyms" -> {}, 
    "title" -> "ExplicitRungeKutta Method for NDSolve", "type" -> "Tutorial", 
    "uri" -> "tutorial/NDSolveExplicitRungeKutta"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:1"->{
  Cell[1328, 44, 93, 2, 70, "Title",
   CellTags->"c:1",
   CellID->1812709110]},
 "s:1"->{
  Cell[1446, 50, 69, 2, 70, "Section",
   CellTags->"s:1",
   CellID->689939648]},
 "ss:1"->{
  Cell[1974, 74, 75, 2, 70, "Subsection",
   CellTags->"ss:1",
   CellID->175801995]},
 "EulersMethod"->{
  Cell[2197, 84, 438, 16, 70, "NumberedEquation",
   CellTags->"EulersMethod",
   CellID->501353605]},
 "ss:2"->{
  Cell[4603, 177, 88, 2, 70, "Subsection",
   CellTags->"ss:2",
   CellID->863140840]},
 "ExplicitMidpointMethod"->{
  Cell[5085, 196, 1150, 39, 70, "NumberedEquation",
   CellTags->"ExplicitMidpointMethod",
   CellID->1322871082]},
 "ERKMethod"->{
  Cell[8752, 339, 1489, 48, 70, "NumberedEquation",
   CellTags->"ERKMethod",
   CellID->194574237]},
 "RowSumConditions"->{
  Cell[10528, 405, 314, 10, 70, "NumberedEquation",
   CellTags->"RowSumConditions",
   CellID->1981247538]},
 "ButcherTable"->{
  Cell[12958, 497, 1133, 32, 70, "NumberedEquation",
   CellTags->"ButcherTable",
   CellID->409178630]},
 "ExplicitMidpointButcherTable"->{
  Cell[14817, 563, 450, 14, 70, "NumberedEquation",
   CellTags->"ExplicitMidpointButcherTable",
   CellID->1739211701]},
 "FSALConditions"->{
  Cell[15610, 592, 437, 14, 70, "NumberedEquation",
   CellTags->"FSALConditions",
   CellID->293358758]},
 "FSALButcherTable"->{
  Cell[16232, 618, 1281, 36, 70, "NumberedEquation",
   CellTags->"FSALButcherTable",
   CellID->1369615083]},
 "EmbeddedButcherTable"->{
  Cell[18884, 705, 1708, 51, 70, "NumberedEquation",
   CellTags->"EmbeddedButcherTable",
   CellID->1874582710]},
 "HigherOrder"->{
  Cell[20659, 761, 456, 16, 70, "NumberedEquation",
   CellTags->"HigherOrder",
   CellID->1349444129]},
 "LowerOrder"->{
  Cell[21118, 779, 507, 18, 70, "NumberedEquation",
   CellTags->"LowerOrder",
   CellID->14497807]},
 "IController"->{
  Cell[24865, 929, 487, 16, 70, "NumberedEquation",
   CellTags->"IController",
   CellID->149660343]},
 "s:2"->{
  Cell[28971, 1092, 65, 2, 70, "Section",
   CellTags->"s:2",
   CellID->1942186939]},
 "s:3"->{
  Cell[38968, 1355, 75, 2, 70, "Section",
   CellTags->"s:3",
   CellID->1456351317]},
 "s:4"->{
  Cell[57781, 1934, 77, 2, 70, "Section",
   CellTags->"s:4",
   CellID->1391088597]},
 "s:5"->{
  Cell[77995, 2639, 71, 2, 70, "Section",
   CellTags->"s:5",
   CellID->551352890]},
 "s:6"->{
  Cell[82958, 2804, 65, 2, 70, "Section",
   CellTags->"s:6",
   CellID->61760426]},
 "DahlquistEquation"->{
  Cell[83752, 2837, 472, 14, 70, "NumberedEquation",
   CellTags->"DahlquistEquation",
   CellID->1160429658]},
 "StiffnessDevice"->{
  Cell[588912, 11254, 583, 17, 70, "NumberedEquation",
   CellTags->"StiffnessDevice",
   CellID->1210923959]},
 "StiffButcherTable"->{
  Cell[591123, 11346, 1339, 38, 70, "NumberedEquation",
   CellTags->"StiffButcherTable",
   CellID->1723376525]},
 "s:7"->{
  Cell[602611, 11748, 80, 2, 70, "Section",
   CellTags->"s:7",
   CellID->1938246688]},
 "PIController"->{
  Cell[612601, 11976, 945, 28, 70, "NumberedEquation",
   CellTags->"PIController",
   CellID->687546269]},
 "StepSizeSafetyFactors"->{
  Cell[633624, 12518, 1071, 32, 70, "NumberedEquation",
   CellTags->"StepSizeSafetyFactors",
   CellID->1498597207]},
 "StepSizeRatioBounds"->{
  Cell[635404, 12580, 391, 11, 70, "NumberedEquation",
   CellTags->"StepSizeRatioBounds",
   CellID->1467635265]},
 "s:8"->{
  Cell[635844, 12597, 72, 2, 70, "Section",
   CellTags->"s:8",
   CellID->1681583708]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:1", 642859, 12811},
 {"s:1", 642951, 12815},
 {"ss:1", 643045, 12819},
 {"EulersMethod", 643151, 12823},
 {"ss:2", 643265, 12827},
 {"ExplicitMidpointMethod", 643382, 12831},
 {"ERKMethod", 643514, 12835},
 {"RowSumConditions", 643639, 12839},
 {"ButcherTable", 643768, 12843},
 {"ExplicitMidpointButcherTable", 643909, 12847},
 {"FSALConditions", 644052, 12851},
 {"FSALButcherTable", 644182, 12855},
 {"EmbeddedButcherTable", 644320, 12859},
 {"HigherOrder", 644453, 12863},
 {"LowerOrder", 644575, 12867},
 {"IController", 644695, 12871},
 {"s:2", 644809, 12875},
 {"s:3", 644906, 12879},
 {"s:4", 645003, 12883},
 {"s:5", 645100, 12887},
 {"s:6", 645196, 12891},
 {"DahlquistEquation", 645305, 12895},
 {"StiffnessDevice", 645439, 12899},
 {"StiffButcherTable", 645575, 12903},
 {"s:7", 645700, 12907},
 {"PIController", 645808, 12911},
 {"StepSizeSafetyFactors", 645944, 12915},
 {"StepSizeRatioBounds", 646089, 12919},
 {"s:8", 646215, 12923}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 703, 17, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1328, 44, 93, 2, 70, "Title",
 CellTags->"c:1",
 CellID->1812709110],
Cell[CellGroupData[{
Cell[1446, 50, 69, 2, 70, "Section",
 CellTags->"s:1",
 CellID->689939648],
Cell[1518, 54, 116, 3, 70, "Text",
 CellID->517979082],
Cell[1637, 59, 312, 11, 70, "Input",
 InitializationCell->True,
 CellID->2013953080],
Cell[CellGroupData[{
Cell[1974, 74, 75, 2, 70, "Subsection",
 CellTags->"ss:1",
 CellID->175801995],
Cell[2052, 78, 142, 4, 70, "Text",
 CellID->2118371868],
Cell[2197, 84, 438, 16, 70, "NumberedEquation",
 CellTags->"EulersMethod",
 CellID->501353605],
Cell[2638, 102, 72, 1, 70, "Text",
 CellID->928734705],
Cell[2713, 105, 356, 10, 70, "Text",
 CellID->701630704],
Cell[3072, 117, 251, 10, 70, "Text",
 CellID->1718414918],
Cell[CellGroupData[{
Cell[3348, 131, 669, 19, 70, "Input",
 CellID->1380692090],
Cell[4020, 152, 534, 19, 36, "Output",
 CellID->47958774]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[4603, 177, 88, 2, 70, "Subsection",
 CellTags->"ss:2",
 CellID->863140840],
Cell[4694, 181, 221, 5, 70, "Text",
 CellID->859415753],
Cell[4918, 188, 164, 6, 70, "Text",
 CellID->748503971],
Cell[5085, 196, 1150, 39, 70, "NumberedEquation",
 CellTags->"ExplicitMidpointMethod",
 CellID->1322871082],
Cell[6238, 237, 261, 10, 70, "Text",
 CellID->667234019],
Cell[6502, 249, 258, 10, 70, "Text",
 CellID->744826097],
Cell[CellGroupData[{
Cell[6785, 263, 672, 19, 70, "Input",
 CellID->1017557097],
Cell[7460, 284, 535, 19, 36, "Output",
 CellID->580441255]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8044, 309, 61, 1, 70, "Subsection",
 CellID->748768759],
Cell[8108, 312, 233, 7, 70, "Text",
 CellID->1811365676],
Cell[8344, 321, 405, 16, 70, "Text",
 CellID->1305450463],
Cell[8752, 339, 1489, 48, 70, "NumberedEquation",
 CellTags->"ERKMethod",
 CellID->194574237],
Cell[10244, 389, 127, 6, 70, "Text",
 CellID->1494356496],
Cell[10374, 397, 151, 6, 70, "Text",
 CellID->467022158],
Cell[10528, 405, 314, 10, 70, "NumberedEquation",
 CellTags->"RowSumConditions",
 CellID->1981247538],
Cell[10845, 417, 218, 5, 70, "Text",
 CellID->1936944985],
Cell[11066, 424, 411, 14, 70, "Text",
 CellID->1512404568],
Cell[11480, 440, 225, 7, 70, "Text",
 CellID->743953168],
Cell[11708, 449, 334, 9, 70, "DisplayMath",
 CellID->341930292],
Cell[12045, 460, 910, 35, 70, "Text",
 CellID->1250344399],
Cell[12958, 497, 1133, 32, 70, "NumberedEquation",
 CellTags->"ButcherTable",
 CellID->409178630],
Cell[14094, 531, 128, 4, 70, "Text",
 CellID->1031869940],
Cell[14225, 537, 316, 10, 70, "Text",
 CellID->1768323481]
}, Open  ]],
Cell[CellGroupData[{
Cell[14578, 552, 49, 1, 70, "Subsection",
 CellID->275648062],
Cell[14630, 555, 184, 6, 70, "Text",
 CellID->1207671936],
Cell[14817, 563, 450, 14, 70, "NumberedEquation",
 CellTags->"ExplicitMidpointButcherTable",
 CellID->1739211701]
}, Open  ]],
Cell[CellGroupData[{
Cell[15304, 582, 54, 1, 70, "Subsection",
 CellID->777001667],
Cell[15361, 585, 246, 5, 70, "Text",
 CellID->2136406641],
Cell[15610, 592, 437, 14, 70, "NumberedEquation",
 CellTags->"FSALConditions",
 CellID->293358758],
Cell[16050, 608, 179, 8, 70, "Text",
 CellID->347932937],
Cell[16232, 618, 1281, 36, 70, "NumberedEquation",
 CellTags->"FSALButcherTable",
 CellID->1369615083],
Cell[17516, 656, 373, 12, 70, "Text",
 CellID->1156499598],
Cell[17892, 670, 473, 14, 70, "Text",
 CellID->1207348691]
}, Open  ]],
Cell[CellGroupData[{
Cell[18402, 689, 84, 1, 70, "Subsection",
 CellID->2118119054],
Cell[18489, 692, 392, 11, 70, "Text",
 CellID->331887032],
Cell[18884, 705, 1708, 51, 70, "NumberedEquation",
 CellTags->"EmbeddedButcherTable",
 CellID->1874582710],
Cell[20595, 758, 61, 1, 70, "Text",
 CellID->715461503],
Cell[20659, 761, 456, 16, 70, "NumberedEquation",
 CellTags->"HigherOrder",
 CellID->1349444129],
Cell[21118, 779, 507, 18, 70, "NumberedEquation",
 CellTags->"LowerOrder",
 CellID->14497807],
Cell[21628, 799, 580, 26, 70, "Text",
 CellID->1464230263],
Cell[22211, 827, 500, 16, 70, "Text",
 CellID->1013326157],
Cell[22714, 845, 1291, 46, 70, "Text",
 CellID->1319948270],
Cell[24008, 893, 335, 11, 70, "DisplayMath",
 CellID->1658804500],
Cell[24346, 906, 309, 10, 70, "Text",
 CellID->360199005]
}, Open  ]],
Cell[CellGroupData[{
Cell[24692, 921, 55, 1, 70, "Subsection",
 CellID->1786970428],
Cell[24750, 924, 112, 3, 70, "Text",
 CellID->1492568390],
Cell[24865, 929, 487, 16, 70, "NumberedEquation",
 CellTags->"IController",
 CellID->149660343],
Cell[25355, 947, 386, 16, 70, "Text",
 CellID->1151891348],
Cell[25744, 965, 148, 4, 70, "Text",
 CellID->89037911],
Cell[25895, 971, 387, 14, 70, "Text",
 CellID->491590635]
}, Open  ]],
Cell[CellGroupData[{
Cell[26319, 990, 50, 1, 70, "Subsection",
 CellID->254752099],
Cell[26372, 993, 123, 4, 70, "Text",
 CellID->498216541],
Cell[26498, 999, 81, 1, 70, "Text",
 CellID->1523460575],
Cell[26582, 1002, 73, 1, 70, "BulletedText",
 CellID->2023075580],
Cell[26658, 1005, 147, 4, 70, "BulletedText",
 CellID->773864604],
Cell[26808, 1011, 519, 19, 70, "BulletedText",
 CellID->583648863],
Cell[CellGroupData[{
Cell[27352, 1034, 244, 8, 70, "BulletedText",
 CellID->1614803634],
Cell[27599, 1044, 347, 11, 70, "Text",
 CellID->1550866271],
Cell[27949, 1057, 374, 11, 70, "Text",
 CellID->572571856],
Cell[28326, 1070, 377, 8, 70, "Text",
 CellID->1908162392],
Cell[28706, 1080, 204, 5, 70, "Text",
 CellID->1898799209]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28971, 1092, 65, 2, 70, "Section",
 CellTags->"s:2",
 CellID->1942186939],
Cell[29039, 1096, 120, 3, 70, "MathCaption",
 CellID->917178342],
Cell[CellGroupData[{
Cell[29184, 1103, 165, 5, 70, "Input",
 CellID->958664252],
Cell[29352, 1110, 1882, 61, 73, "Output",
 CellID->489867450]
}, Open  ]],
Cell[31249, 1174, 106, 2, 70, "MathCaption",
 CellID->1249065375],
Cell[CellGroupData[{
Cell[31380, 1180, 232, 7, 70, "Input",
 CellID->685763314],
Cell[31615, 1189, 967, 34, 54, "Output",
 CellID->57136273]
}, Open  ]],
Cell[32597, 1226, 97, 1, 70, "MathCaption",
 CellID->451025856],
Cell[32697, 1229, 250, 7, 70, "Input",
 CellID->700261121],
Cell[32950, 1238, 73, 1, 70, "MathCaption",
 CellID->1945030464],
Cell[CellGroupData[{
Cell[33048, 1243, 263, 8, 28, "Input",
 CellID->1464352014],
Cell[33314, 1253, 5605, 96, 230, "Output",
 Evaluatable->False,
 CellID->32302612]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[38968, 1355, 75, 2, 70, "Section",
 CellTags->"s:3",
 CellID->1456351317],
Cell[39046, 1359, 252, 9, 70, "Text",
 CellID->1824551019],
Cell[39301, 1370, 120, 3, 70, "MathCaption",
 CellID->1659696171],
Cell[CellGroupData[{
Cell[39446, 1377, 173, 4, 70, "Input",
 CellID->801436059],
Cell[39622, 1383, 736, 27, 51, "Output",
 CellID->635475014]
}, Open  ]],
Cell[40373, 1413, 169, 4, 70, "Text",
 CellID->664103872],
Cell[CellGroupData[{
Cell[40567, 1421, 51, 1, 70, "Subsection",
 CellID->890473970],
Cell[40621, 1424, 349, 10, 70, "Text",
 CellID->318318349],
Cell[40973, 1436, 173, 5, 70, "BulletedText",
 CellID->918855971],
Cell[CellGroupData[{
Cell[41171, 1445, 189, 6, 70, "BulletedText",
 CellID->1388690512],
Cell[41363, 1453, 152, 5, 70, "MathCaption",
 CellID->1841778925],
Cell[41518, 1460, 1176, 30, 70, "Input",
 CellID->794447904]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[42743, 1496, 59, 1, 70, "Subsection",
 CellID->15071468],
Cell[42805, 1499, 212, 5, 70, "Text",
 CellID->277944140],
Cell[43020, 1506, 348, 11, 70, "Text",
 CellID->261767890],
Cell[43371, 1519, 254, 6, 70, "MathCaption",
 CellID->1565937739],
Cell[43628, 1527, 512, 15, 70, "Input",
 CellID->850515997]
}, Open  ]],
Cell[CellGroupData[{
Cell[44177, 1547, 66, 1, 70, "Subsection",
 CellID->22446634],
Cell[44246, 1550, 220, 6, 70, "Text",
 CellID->1317253300],
Cell[44469, 1558, 267, 8, 70, "Text",
 CellID->645219288]
}, Open  ]],
Cell[CellGroupData[{
Cell[44773, 1571, 51, 1, 70, "Subsection",
 CellID->939624976],
Cell[44827, 1574, 171, 4, 70, "Text",
 CellID->128433904],
Cell[45001, 1580, 285, 9, 70, "MathCaption",
 CellID->410557986],
Cell[45289, 1591, 833, 25, 70, "Input",
 CellID->41206703],
Cell[46125, 1618, 146, 4, 70, "MathCaption",
 CellID->814197219],
Cell[46274, 1624, 230, 7, 70, "Input",
 CellID->891358188],
Cell[46507, 1633, 74, 1, 70, "MathCaption",
 CellID->492232560],
Cell[CellGroupData[{
Cell[46606, 1638, 406, 11, 70, "Input",
 CellID->1514458340],
Cell[47015, 1651, 1436, 43, 228, "Output",
 CellID->182305699]
}, Open  ]],
Cell[48466, 1697, 235, 5, 70, "Text",
 CellID->1956584574]
}, Open  ]],
Cell[CellGroupData[{
Cell[48738, 1707, 52, 1, 70, "Subsection",
 CellID->1387184084],
Cell[48793, 1710, 231, 5, 70, "Text",
 CellID->2145256321],
Cell[49027, 1717, 151, 4, 70, "Text",
 CellID->1633651945],
Cell[49181, 1723, 193, 5, 70, "Text",
 CellID->899396607],
Cell[49377, 1730, 285, 9, 70, "MathCaption",
 CellID->799208733],
Cell[49665, 1741, 785, 24, 70, "Input",
 CellID->1460379660],
Cell[50453, 1767, 181, 5, 70, "MathCaption",
 CellID->1563500479],
Cell[50637, 1774, 524, 15, 70, "Input",
 CellID->1689682703],
Cell[51164, 1791, 384, 7, 70, "MathCaption",
 CellID->930889567],
Cell[CellGroupData[{
Cell[51573, 1802, 1070, 30, 65, "Input",
 CellID->1472993286],
Cell[52646, 1834, 4711, 81, 141, "Output",
 Evaluatable->False,
 CellID->395729241]
}, Open  ]],
Cell[57372, 1918, 237, 5, 70, "Text",
 CellID->945961035],
Cell[57612, 1925, 120, 3, 70, "Text",
 CellID->1750937427]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[57781, 1934, 77, 2, 70, "Section",
 CellTags->"s:4",
 CellID->1391088597],
Cell[57861, 1938, 163, 5, 70, "Text",
 CellID->364415127],
Cell[58027, 1945, 120, 3, 70, "Text",
 CellID->453969776],
Cell[58150, 1950, 86, 1, 70, "BulletedText",
 CellID->1413535231],
Cell[58239, 1953, 124, 3, 70, "BulletedText",
 CellID->1681668427],
Cell[58366, 1958, 76, 1, 70, "BulletedText",
 CellID->1099189064],
Cell[CellGroupData[{
Cell[58467, 1963, 74, 1, 70, "Subsection",
 CellID->205676559],
Cell[58544, 1966, 217, 6, 70, "MathCaption",
 CellID->1810102332],
Cell[58764, 1974, 995, 33, 70, "Input",
 CellID->1509747051],
Cell[59762, 2009, 219, 5, 70, "Text",
 CellID->246883263],
Cell[59984, 2016, 84, 1, 70, "MathCaption",
 CellID->476815903],
Cell[CellGroupData[{
Cell[60093, 2021, 486, 13, 70, "Input",
 CellID->253621591],
Cell[60582, 2036, 968, 34, 54, "Output",
 CellID->59681361]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[61599, 2076, 48, 1, 70, "Subsection",
 CellID->1991274243],
Cell[61650, 2079, 116, 3, 70, "Text",
 CellID->942526595],
Cell[61769, 2084, 262, 8, 70, "Text",
 CellID->548216749],
Cell[62034, 2094, 128, 3, 70, "MathCaption",
 CellID->1230974679],
Cell[62165, 2099, 1447, 49, 70, "Input",
 CellID->940299269],
Cell[63615, 2150, 245, 8, 70, "Text",
 CellID->34385046],
Cell[63863, 2160, 136, 4, 70, "Text",
 CellID->2130371287]
}, Open  ]],
Cell[CellGroupData[{
Cell[64036, 2169, 63, 1, 70, "Subsection",
 CellID->1739508002],
Cell[64102, 2172, 266, 8, 70, "Text",
 CellID->1494764706],
Cell[64371, 2182, 183, 4, 70, "Text",
 CellID->1546388118],
Cell[64557, 2188, 128, 3, 70, "MathCaption",
 CellID->1281707057],
Cell[64688, 2193, 2418, 74, 70, "Input",
 CellID->1125979068],
Cell[67109, 2269, 189, 4, 70, "Text",
 CellID->1881260921],
Cell[67301, 2275, 528, 14, 70, "Text",
 CellID->65112977]
}, Open  ]],
Cell[CellGroupData[{
Cell[67866, 2294, 65, 1, 70, "Subsection",
 CellID->451488706],
Cell[67934, 2297, 293, 9, 70, "Text",
 CellID->863097980],
Cell[68230, 2308, 128, 3, 70, "MathCaption",
 CellID->1011074012],
Cell[68361, 2313, 2922, 89, 70, "Input",
 CellID->54942485],
Cell[71286, 2404, 285, 8, 70, "Text",
 CellID->106702929],
Cell[71574, 2414, 300, 6, 70, "MathCaption",
 CellID->1375489139],
Cell[CellGroupData[{
Cell[71899, 2424, 450, 12, 70, "Input",
 CellID->508912261],
Cell[72352, 2438, 968, 34, 54, "Output",
 CellID->26032513]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[73369, 2478, 59, 1, 70, "Subsection",
 CellID->909673919],
Cell[73431, 2481, 109, 3, 70, "Text",
 CellID->1608101326],
Cell[73543, 2486, 200, 5, 70, "MathCaption",
 CellID->1725914720],
Cell[73746, 2493, 490, 12, 70, "Input",
 CellID->572292603],
Cell[74239, 2507, 98, 1, 70, "MathCaption",
 CellID->970364729],
Cell[74340, 2510, 402, 10, 70, "Input",
 CellID->1972797983],
Cell[74745, 2522, 123, 3, 70, "MathCaption",
 CellID->1049108506],
Cell[74871, 2527, 441, 12, 70, "Input",
 CellID->1223986320],
Cell[75315, 2541, 113, 3, 70, "MathCaption",
 CellID->564430158],
Cell[75431, 2546, 467, 15, 70, "Input",
 CellID->892899101],
Cell[75901, 2563, 145, 4, 70, "MathCaption",
 CellID->19029267],
Cell[76049, 2569, 226, 7, 70, "Input",
 CellID->1112646907],
Cell[76278, 2578, 75, 1, 70, "MathCaption",
 CellID->2123873781],
Cell[CellGroupData[{
Cell[76378, 2583, 404, 11, 70, "Input",
 CellID->959328536],
Cell[76785, 2596, 1016, 30, 114, "Output",
 CellID->314363219]
}, Open  ]],
Cell[77816, 2629, 130, 4, 70, "Text",
 CellID->2106761420]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[77995, 2639, 71, 2, 70, "Section",
 CellTags->"s:5",
 CellID->551352890],
Cell[78069, 2643, 167, 4, 70, "Text",
 CellID->494313012],
Cell[78239, 2649, 326, 6, 70, "MathCaption",
 CellID->1505665890],
Cell[78568, 2657, 280, 7, 70, "Input",
 CellID->1833059948],
Cell[78851, 2666, 123, 3, 70, "MathCaption",
 CellID->1620292080],
Cell[78977, 2671, 1762, 54, 70, "Input",
 CellID->1239443164],
Cell[80742, 2727, 382, 7, 70, "Text",
 CellID->1898172925],
Cell[81127, 2736, 195, 4, 70, "MathCaption",
 CellID->2071780786],
Cell[CellGroupData[{
Cell[81347, 2744, 267, 7, 70, "Input",
 CellID->2102541144],
Cell[81617, 2753, 969, 34, 54, "Output",
 CellID->105002239]
}, Open  ]],
Cell[82601, 2790, 320, 9, 70, "Text",
 CellID->2126723272]
}, Open  ]],
Cell[CellGroupData[{
Cell[82958, 2804, 65, 2, 70, "Section",
 CellTags->"s:6",
 CellID->61760426],
Cell[83026, 2808, 139, 4, 70, "Text",
 CellID->2137252429],
Cell[83168, 2814, 169, 4, 70, "Text",
 CellID->488151885],
Cell[83340, 2820, 179, 4, 70, "Text",
 CellID->1581998586],
Cell[CellGroupData[{
Cell[83544, 2828, 58, 1, 70, "Subsection",
 CellID->899442700],
Cell[83605, 2831, 144, 4, 70, "Text",
 CellID->2096253211],
Cell[83752, 2837, 472, 14, 70, "NumberedEquation",
 CellTags->"DahlquistEquation",
 CellID->1160429658],
Cell[84227, 2853, 437, 16, 70, "Text",
 CellID->1471735303],
Cell[84667, 2871, 316, 8, 70, "Text",
 CellID->919134292],
Cell[84986, 2881, 143, 4, 70, "MathCaption",
 CellID->985743993],
Cell[CellGroupData[{
Cell[85154, 2889, 224, 5, 70, "Input",
 CellID->373360161],
Cell[85381, 2896, 466, 16, 54, "Output",
 CellID->63142193]
}, Open  ]],
Cell[85862, 2915, 309, 8, 70, "Text",
 CellID->1550174151],
Cell[86174, 2925, 176, 4, 70, "MathCaption",
 CellID->1266259262],
Cell[86353, 2931, 152, 5, 70, "Input",
 CellID->14666232],
Cell[86508, 2938, 259, 10, 70, "MathCaption",
 CellID->515337410],
Cell[CellGroupData[{
Cell[86792, 2952, 155, 4, 70, "Input",
 CellID->1628247325],
Cell[86950, 2958, 500512, 8237, 70, "Output",
 CellID->56285197]
}, Open  ]],
Cell[587477, 11198, 529, 17, 70, "Text",
 CellID->1662369364],
Cell[588009, 11217, 320, 11, 70, "Text",
 CellID->1201927167]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[588378, 11234, 59, 1, 70, "Section",
 CellID->1525347659],
Cell[588440, 11237, 313, 9, 70, "Text",
 CellID->9198728],
Cell[588756, 11248, 153, 4, 70, "Text",
 CellID->560109410],
Cell[588912, 11254, 583, 17, 70, "NumberedEquation",
 CellTags->"StiffnessDevice",
 CellID->1210923959],
Cell[589498, 11273, 286, 14, 70, "Text",
 CellID->1629490808],
Cell[589787, 11289, 421, 17, 70, "Text",
 CellID->1799552574],
Cell[590211, 11308, 154, 4, 70, "Text",
 CellID->961613236],
Cell[590368, 11314, 306, 10, 70, "Text",
 CellID->1553787946],
Cell[590677, 11326, 443, 18, 70, "Text",
 CellID->1754610070],
Cell[591123, 11346, 1339, 38, 70, "NumberedEquation",
 CellTags->"StiffButcherTable",
 CellID->1723376525],
Cell[592465, 11386, 224, 10, 70, "Text",
 CellID->1533027570],
Cell[592692, 11398, 280, 7, 70, "Text",
 CellID->1816370496],
Cell[592975, 11407, 267, 9, 70, "Text",
 CellID->1713145306],
Cell[CellGroupData[{
Cell[593267, 11420, 51, 1, 70, "Subsection",
 CellID->2021179456],
Cell[593321, 11423, 95, 1, 70, "MathCaption",
 CellID->1250885435],
Cell[593419, 11426, 190, 6, 70, "Input",
 CellID->465104358],
Cell[593612, 11434, 124, 3, 70, "MathCaption",
 CellID->1969949019],
Cell[593739, 11439, 149, 4, 70, "MathCaption",
 CellID->263831445],
Cell[CellGroupData[{
Cell[593913, 11447, 201, 6, 70, "Input",
 CellID->1936839661],
Cell[594117, 11455, 328, 7, 70, "Message",
 CellID->185903983],
Cell[594448, 11464, 1397, 47, 71, "Output",
 CellID->277105984]
}, Open  ]],
Cell[595860, 11514, 252, 6, 70, "MathCaption",
 CellID->1249575439],
Cell[CellGroupData[{
Cell[596137, 11524, 489, 13, 70, "Input",
 CellID->162716174],
Cell[596629, 11539, 333, 7, 70, "Message",
 CellID->222257229],
Cell[596965, 11548, 258, 6, 70, "Message",
 CellID->515687557]
}, Open  ]],
Cell[597238, 11557, 269, 5, 70, "Text",
 CellID->737236628],
Cell[597510, 11564, 212, 5, 70, "MathCaption",
 CellID->1428975437],
Cell[CellGroupData[{
Cell[597747, 11573, 282, 9, 70, "Input",
 CellID->751902860],
Cell[598032, 11584, 564, 18, 39, "Output",
 CellID->98308044]
}, Open  ]],
Cell[598611, 11605, 166, 4, 70, "Text",
 CellID->524141184],
Cell[598780, 11611, 125, 3, 70, "MathCaption",
 CellID->1725897629],
Cell[598908, 11616, 618, 20, 70, "Input",
 CellID->682489069],
Cell[599529, 11638, 207, 6, 70, "MathCaption",
 CellID->1726068352],
Cell[CellGroupData[{
Cell[599761, 11648, 380, 10, 70, "Input",
 CellID->1103761640],
Cell[600144, 11660, 328, 7, 70, "Message",
 CellID->57312333],
Cell[600475, 11669, 1400, 47, 71, "Output",
 CellID->952136848]
}, Open  ]],
Cell[601890, 11719, 409, 15, 70, "Text",
 CellID->167201951],
Cell[602302, 11736, 260, 6, 70, "Text",
 CellID->173694957]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[602611, 11748, 80, 2, 70, "Section",
 CellTags->"s:7",
 CellID->1938246688],
Cell[602694, 11752, 239, 7, 70, "Text",
 CellID->1128815187],
Cell[602936, 11761, 83, 1, 70, "MathCaption",
 CellID->1727482903],
Cell[603022, 11764, 191, 6, 70, "Input",
 CellID->1545373110],
Cell[603216, 11772, 176, 4, 70, "MathCaption",
 CellID->704708064],
Cell[603395, 11778, 399, 10, 70, "Input",
 CellID->1113026826],
Cell[603797, 11790, 190, 6, 70, "MathCaption",
 CellID->1214259959],
Cell[CellGroupData[{
Cell[604012, 11800, 288, 8, 47, "Input",
 CellID->270369723],
Cell[604303, 11810, 7172, 122, 128, "Output",
 Evaluatable->False,
 CellID->175811298]
}, Open  ]],
Cell[611490, 11935, 222, 5, 70, "Text",
 CellID->317661746],
Cell[611715, 11942, 95, 1, 70, "Text",
 CellID->735474081],
Cell[611813, 11945, 162, 4, 70, "Text",
 CellID->809188005],
Cell[611978, 11951, 159, 4, 70, "Text",
 CellID->581493191],
Cell[CellGroupData[{
Cell[612162, 11959, 57, 1, 70, "Subsection",
 CellID->524830095],
Cell[612222, 11962, 204, 6, 70, "Text",
 CellID->1550787669],
Cell[612429, 11970, 169, 4, 70, "Text",
 CellID->1831068630],
Cell[612601, 11976, 945, 28, 70, "NumberedEquation",
 CellTags->"PIController",
 CellID->687546269],
Cell[613549, 12006, 123, 4, 70, "Text",
 CellID->140540650],
Cell[613675, 12012, 255, 9, 70, "Text",
 CellID->335850925],
Cell[613933, 12023, 226, 8, 70, "DisplayMath",
 CellID->2116521706],
Cell[614162, 12033, 528, 22, 70, "Text",
 CellID->892443510],
Cell[614693, 12057, 586, 19, 70, "Text",
 CellID->1503454278]
}, Open  ]],
Cell[615294, 12079, 50, 1, 70, "Subsection",
 CellID->493179243],
Cell[CellGroupData[{
Cell[615369, 12084, 55, 1, 70, "Subsection",
 CellID->470857491],
Cell[615427, 12087, 214, 7, 70, "Text",
 CellID->38459699],
Cell[615644, 12096, 106, 3, 70, "Text",
 CellID->71763947],
Cell[615753, 12101, 366, 14, 70, "DisplayMath",
 CellID->1574619131],
Cell[616122, 12117, 86, 1, 70, "MathCaption",
 CellID->730147496],
Cell[616211, 12120, 584, 14, 70, "Input",
 CellID->210458061],
Cell[616798, 12136, 148, 4, 70, "MathCaption",
 CellID->1741821082],
Cell[CellGroupData[{
Cell[616971, 12144, 291, 8, 47, "Input",
 CellID->903842453],
Cell[617265, 12154, 2478, 45, 125, "Output",
 Evaluatable->False,
 CellID->93585090]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[619792, 12205, 58, 1, 70, "Subsection",
 CellID->723152621],
Cell[619853, 12208, 320, 10, 70, "Text",
 CellID->483841708],
Cell[620176, 12220, 116, 3, 70, "MathCaption",
 CellID->1546594211],
Cell[620295, 12225, 196, 6, 70, "Input",
 CellID->1427860686],
Cell[CellGroupData[{
Cell[620516, 12235, 289, 8, 47, "Input",
 CellID->1596868534],
Cell[620808, 12245, 5431, 93, 133, "Output",
 Evaluatable->False,
 CellID->507999994]
}, Open  ]],
Cell[626254, 12341, 117, 3, 70, "MathCaption",
 CellID->1698230409],
Cell[CellGroupData[{
Cell[626396, 12348, 291, 8, 47, "Input",
 CellID->436107520],
Cell[626690, 12358, 5394, 92, 136, "Output",
 Evaluatable->False,
 CellID->618346149]
}, Open  ]],
Cell[632099, 12453, 209, 7, 70, "Text",
 CellID->286220842],
Cell[632311, 12462, 201, 8, 70, "BulletedText",
 CellID->1245295526],
Cell[632515, 12472, 202, 8, 70, "BulletedText",
 CellID->1512525460]
}, Open  ]],
Cell[CellGroupData[{
Cell[632754, 12485, 53, 1, 70, "Subsection",
 CellID->811050937],
Cell[632810, 12488, 308, 8, 70, "Text",
 CellID->648599547],
Cell[633121, 12498, 500, 18, 70, "Text",
 CellID->1196110442],
Cell[633624, 12518, 1071, 32, 70, "NumberedEquation",
 CellTags->"StepSizeSafetyFactors",
 CellID->1498597207],
Cell[634698, 12552, 286, 11, 70, "Text",
 CellID->789140197],
Cell[634987, 12565, 414, 13, 70, "Text",
 CellID->1531961367],
Cell[635404, 12580, 391, 11, 70, "NumberedEquation",
 CellTags->"StepSizeRatioBounds",
 CellID->1467635265]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[635844, 12597, 72, 2, 70, "Section",
 CellTags->"s:8",
 CellID->1681583708],
Cell[635919, 12601, 2035, 56, 70, "DefinitionBox3Col",
 CellID->308103003],
Cell[637957, 12659, 139, 5, 70, "Caption",
 CellID->1960814508],
Cell[638099, 12666, 339, 9, 70, "Text",
 CellID->107132032],
Cell[638441, 12677, 351, 10, 70, "Text",
 CellID->501489466],
Cell[638795, 12689, 308, 11, 70, "Text",
 CellID->12788916],
Cell[639106, 12702, 613, 21, 70, "Text",
 CellID->623800798],
Cell[639722, 12725, 382, 12, 70, "Text",
 CellID->6412433],
Cell[CellGroupData[{
Cell[640129, 12741, 71, 1, 70, "RelatedTutorialsSection",
 CellID->77323823],
Cell[640203, 12744, 326, 9, 70, "RelatedTutorials",
 CellID->604721514]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[640568, 12758, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

