(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    123865,       3228]
NotebookOptionsPosition[    111577,       2845]
NotebookOutlinePosition[    115376,       2936]
CellTagsIndexPosition[    115226,       2928]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[Cell[
    BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"Automatic" :> Documentation`HelpLookup["paclet:ref/Automatic"], 
         "Band" :> Documentation`HelpLookup["paclet:ref/Band"], "LinearSolve" :> 
         Documentation`HelpLookup["paclet:ref/LinearSolve"]}, #, 2]& , 
       Evaluator -> Automatic], {1->"\<\"Automatic\"\>", 2->"\<\"Band\"\>", 
      3->"\<\"LinearSolve\"\>"}, "\<\"functions\"\>", 
      StyleBox["\<\"functions\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["IDA Method for NDSolve", "Title",
 CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"},
 CellID->1283904336],

Cell[TextData[{
 "The ",
 StyleBox["IDA", "MR"],
 " package is part of the ",
 ButtonBox["SUNDIALS",
  BaseStyle->"Hyperlink",
  ButtonData:>{
    URL["http://www.llnl.gov/CASC/sundials/"], None}],
 " (SUite of Nonlinear and DIfferential/ALgebraic equation Solvers) developed \
at the Center for Applied Scientific Computing of Lawrence Livermore National \
Laboratory. As described in the IDA user guide [",
 ButtonBox["HT99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#17230"],
 "], \[OpenCurlyDoubleQuote]IDA is a general purpose solver for the initial \
value problem for systems of differential-algebraic equations (DAEs). The \
name IDA stands for Implicit Differential-Algebraic solver. IDA is based on \
DASPK ...\[CloseCurlyDoubleQuote] ",
 ButtonBox["DASPK",
  BaseStyle->"Hyperlink",
  ButtonData:>{
    URL["http://www.netlib.org/ode/daspk.tgz"], None}],
 " [",
 ButtonBox["BHP94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#19530"],
 "], [",
 ButtonBox["BHP98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#16131"],
 "] is a FORTRAN code for solving large scale differential-algebraic \
systems."
}], "Text",
 CellID->544144180],

Cell[TextData[{
 "In ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", an interface has been provided to the ",
 StyleBox["IDA", "MR"],
 " package so that rather than needing to write a function in C for \
evaluating the residual and compiling the program, ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " generates the function automatically from the equations you input to ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->1870584582],

Cell[TextData[{
 "IDA solves the system (1) with Backward Differentiation Formula (BDF) \
methods of orders 1 through 5, implemented using a variable-step form. The \
BDF of order ",
 StyleBox["k",
  FontSlant->"Italic"],
 " is at time ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["t", "n"], "=", " ", 
    RowBox[{
     SubscriptBox["t", 
      RowBox[{"n", "-", "1"}]], " ", "+", " ", 
     SubscriptBox["h", "n"]}]}], TraditionalForm]]],
 " is given by the formula"
}], "Text",
 CellID->346288529],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], "k"], 
    RowBox[{
     SubscriptBox["a", 
      RowBox[{"n", ",", "i"}]], " ", 
     SubscriptBox["x", 
      RowBox[{"n", "-", "i"}]]}]}], "=", " ", 
   RowBox[{
    SubscriptBox["h", "n"], " ", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["x", "n"], "\[Prime]",
      MultilineFunction->None], "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->1056205345],

Cell[TextData[{
 "The coefficients ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["a", 
    RowBox[{"n", ",", "i"}]], TraditionalForm]], "DisplayMath"],
 " depend on the order ",
 StyleBox["k",
  FontSlant->"Italic"],
 " and past step sizes. Applying the BDF to the DAE (1) gives a system of \
nonlinear equations to solve:"
}], "Text",
 CellID->998665234],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"F", "(", 
    RowBox[{
     SubscriptBox["t", "n"], ",", " ", 
     SubscriptBox["x", "n"], ",", " ", 
     RowBox[{
      FractionBox["1", 
       SubscriptBox["h", "n"]], 
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        RowBox[{"i", "=", "1"}], "k"], 
       RowBox[{
        SubscriptBox["a", 
         RowBox[{"n", ",", "i"}]], " ", 
        SubscriptBox["x", 
         RowBox[{"n", "-", "i"}]]}]}]}]}], ")"}], " ", "=", " ", "0."}], 
  TraditionalForm]], "DisplayMath",
 CellID->1077267824],

Cell["\<\
The solution of the system is achieved by Newton-type methods, typically \
using an approximation to the Jacobian\
\>", "Text",
 CellID->993285444],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"J", " ", "=", " ", 
    RowBox[{
     FractionBox[
      RowBox[{"\[PartialD]", "F"}], 
      RowBox[{"\[PartialD]", "x"}]], "+", " ", 
     RowBox[{
      SubscriptBox["c", "n"], 
      FractionBox[
       RowBox[{"\[PartialD]", "F"}], 
       RowBox[{"\[PartialD]", 
        RowBox[{"x", "'"}]}]]}]}]}], " ", ",", " ", 
   RowBox[{
    RowBox[{"where", " ", 
     SubscriptBox["c", "n"]}], " ", "=", " ", 
    RowBox[{
     FractionBox[
      SubscriptBox["\[Alpha]", 
       RowBox[{"n", ",", "0"}]], 
      SubscriptBox["h", "n"]], "."}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellID->1650775136],

Cell[TextData[{
 " \[OpenCurlyDoubleQuote]Its [IDAs] most notable feature is that, in the \
solution of the underlying nonlinear system at each time step, it offers a \
choice of Newton/direct methods or an Inexact Newton/Krylov (iterative) \
method.\[CloseCurlyDoubleQuote] [",
 ButtonBox["HT99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#17230"],
 "] In ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ", you can access these solvers using method options or use the default ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " ",
 Cell[BoxData[
  ButtonBox["LinearSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
 ", which switches automatically to direct sparse solvers for large \
problems."
}], "Text",
 CellID->1129336518],

Cell[TextData[{
 "At each step of the solution, IDA computes an estimate ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["E", "n"], TraditionalForm]]],
 " of the local truncation error and the step size and order are chosen so \
that the weighted norm ",
 Cell[BoxData[
  RowBox[{"Norm", "[", 
   RowBox[{
    SubscriptBox["E", "n"], "/", 
    SubscriptBox["w", "n"]}], "]"}]], "Input",
  FontWeight->"Plain"],
 "is less than 1. The ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["j", "th"], TraditionalForm]]],
 "component, ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["w",
    FontSlant->"Italic"], 
   RowBox[{
    StyleBox["n",
     FontSlant->"Italic"], ",", 
    StyleBox["j",
     FontSlant->"Italic"]}]]], "Input",
  FontWeight->"Plain"],
 ", of ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["w",
    FontSlant->"Italic"], 
   StyleBox["n",
    FontSlant->"Italic"]]], "Input",
  FontWeight->"Plain"],
 " is given by"
}], "Text",
 CellID->1812989206],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["w", 
    RowBox[{"n", ",", "j"}]], "=", 
   RowBox[{
    FractionBox["1", 
     RowBox[{
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "prec"}]], " ", 
       RowBox[{"\[LeftBracketingBar]", 
        SubscriptBox["x", 
         RowBox[{"n", ",", "j"}]], "\[RightBracketingBar]"}]}], "+", 
      SuperscriptBox["10", 
       RowBox[{"-", "acc"}]]}]], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->1870712677],

Cell[TextData[{
 "The values ",
 StyleBox["prec", "MR"],
 " and ",
 StyleBox["acc", "MR"],
 " are taken from the ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " settings for the ",
 StyleBox["PrecisionGoal->prec", "MR"],
 " and ",
 StyleBox["AccuracyGoal->acc", "MR"],
 ". "
}], "Text",
 CellID->1530839459],

Cell[TextData[{
 "Because IDA provides a great deal of flexibility, particularly in the way \
nonlinear equations are solved, there are a number of method options which \
allow you to control how this is done. You can use the method options to IDA \
by giving ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " the option ",
 StyleBox["Method->{IDA, ", "MR"],
 StyleBox["ida method options", "MR",
  FontSlant->"Italic"],
 StyleBox["}", "MR"],
 "."
}], "Text",
 CellID->78251959],

Cell[TextData[{
 "The options for the ",
 StyleBox["IDA", "MR"],
 " method are associated with the symbol ",
 StyleBox["IDA", "MR"],
 " in the ",
 StyleBox["NDSolve`", "MR"],
 " context."
}], "MathCaption",
 CellID->215238697],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Options", "[", "NDSolve`IDA", "]"}]], "Input",
 CellLabel->"In[66]:=",
 CellID->326296895],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\<\"MaxDifferenceOrder\"\>", "\[Rule]", "5"}], ",", 
   RowBox[{"\<\"ImplicitSolver\"\>", "\[Rule]", "\<\"Newton\"\>"}]}], 
  "}"}]], "Output",
 ImageSize->{335, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[66]=",
 CellID->11683416]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["IDA method option name", "TableHeader"], Cell[
    "default value", "TableHeader"], " "},
   {"\"\<ImplicitSolver\>\"", "\"\<Newton\>\"", Cell[
    "How to solve the implicit equations. ", "TableText"]},
   {"\"\<MaxDifferenceOrder\>\"", "5", Cell[
    "The maximum order BDF to use. ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->694232409],

Cell[TextData[{
 Cell[BoxData["IDA"], "InlineFormula"],
 " method options."
}], "Caption",
 CellID->2135801483],

Cell[TextData[{
 "When strict accuracy of intermediate values computed with the ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object returned from ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " is important, you will want to use the ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " method option setting ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["InterpolationOrder",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/InterpolationOrder"], "->", 
   ButtonBox["All",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/All"]}]], "InlineFormula"],
 " that uses interpolation based on the order of the method, sometimes called \
dense output, to represent the solution between times steps. By default \
NDSolve stores a minimal amount of data to represent the solution well enough \
for graphical purposes. Keeping the amount of data small saves on both memory \
and time for more complicated solutions. "
}], "Text",
 CellID->555629979],

Cell[TextData[{
 "As an example which highlights the difference between minimal output and \
full method interpolation order, consider tracking a quantity, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"f", "(", "t", ")"}], "=", " ", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"x", "(", "t", ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{"y", "(", "t", ")"}], "2"]}]}], TraditionalForm]]],
 " derived from the solution of a simple linear equation where the exact \
solution can be computed using ",
 StyleBox["DSolve", "MR"],
 "."
}], "Text",
 CellID->121628007],

Cell["\<\
This defines the function f giving the quantity as a function of time based \
on solutions x[t] and y[t].\
\>", "MathCaption",
 CellID->140795431],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "t_", "]"}], "  ", ":=", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"x", "[", "t", "]"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"y", "[", "t", "]"}], "2"]}]}], ";"}]], "Input",
 CellLabel->"In[67]:=",
 CellID->1558923092],

Cell["\<\
This defines the linear equations along with initial conditions.\
\>", "MathCaption",
 CellID->1696791596],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eqns", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"x", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], "-", 
       RowBox[{"2", " ", 
        RowBox[{"y", "[", "t", "]"}]}]}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "'"}], "[", "t", "]"}], "\[Equal]", 
      RowBox[{
       RowBox[{"x", "[", "t", "]"}], "+", 
       RowBox[{"y", "[", "t", "]"}]}]}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"ics", " ", "=", " ", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}], ",", " ", 
     RowBox[{
      RowBox[{"y", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[68]:=",
 CellID->1675815117],

Cell[TextData[{
 "The exact value of f as a function of time can be computed symbolically \
using ",
 StyleBox["DSolve", "MR"],
 "."
}], "MathCaption",
 CellID->315677786],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["f", "exact"], "[", "t_", "]"}], " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{
    RowBox[{"f", "[", "t", "]"}], "/.", 
    RowBox[{"DSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"eqns", ",", " ", "ics"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y"}], "}"}], ",", "t"}], "]"}]}], "]"}]}]], "Input",\

 CellLabel->"In[70]:=",
 CellID->1027584235],

Cell[BoxData[
 RowBox[{
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"2", " ", "t"}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Cos", "[", 
       RowBox[{
        SqrtBox["2"], " ", "t"}], "]"}], "-", 
      RowBox[{
       SqrtBox["2"], " ", 
       RowBox[{"Sin", "[", 
        RowBox[{
         SqrtBox["2"], " ", "t"}], "]"}]}]}], ")"}], "2"]}], "+", 
  RowBox[{
   FractionBox["1", "4"], " ", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"2", " ", "t"}]], " ", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", " ", 
       RowBox[{"Cos", "[", 
        RowBox[{
         SqrtBox["2"], " ", "t"}], "]"}]}], "+", 
      RowBox[{
       SqrtBox["2"], " ", 
       RowBox[{"Sin", "[", 
        RowBox[{
         SqrtBox["2"], " ", "t"}], "]"}]}]}], ")"}], "2"]}]}]], "Output",
 ImageSize->{480, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[70]=",
 CellID->139758597]
}, Open  ]],

Cell["\<\
The exact solution will be compared with solutions computed with and without \
dense output.\
\>", "Text",
 CellID->1097972321],

Cell["\<\
A simple way to track the quantity is to create a function which derives it \
from the numerical solution of the differential equation.\
\>", "MathCaption",
 CellID->1178722026],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"f1", "[", "t_", "]"}], " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{
    RowBox[{"f", "[", "t", "]"}], " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"eqns", ",", " ", "ics"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}]], "Input",\

 CellLabel->"In[71]:=",
 CellID->977014750],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "t", "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "t", "]"}], "2"]}]], "Output",
 ImageSize->{577, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[71]=",
 CellID->325918764]
}, Open  ]],

Cell["It can also be computed with dense output.", "MathCaption",
 CellID->1722866206],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["f1", "dense"], "[", "t_", "]"}], " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{
    RowBox[{"f", "[", "t", "]"}], " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"eqns", ",", " ", "ics"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", " ", 
      RowBox[{"InterpolationOrder", "\[Rule]", "All"}]}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[72]:=",
 CellID->700115685],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "t", "]"}], "2"], "+", 
  SuperscriptBox[
   RowBox[{
    TagBox[
     RowBox[{"InterpolatingFunction", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], 
      "]"}],
     False,
     Editable->False], "[", "t", "]"}], "2"]}]], "Output",
 ImageSize->{577, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[72]=",
 CellID->483959973]
}, Open  ]],

Cell["\<\
This shows a plot showing the error in the two computed solutions. The \
computed solution at the time steps are indicated by black dots. The default \
output error is shown in gray and the dense output error in black.\
\>", "MathCaption",
 CellID->631587903],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`InterpolatingFunctionAnatomy`\>\"",
    "]"}], ";", 
  RowBox[{"t1", " ", "=", 
   RowBox[{
    RowBox[{"Cases", "[", 
     RowBox[{
      RowBox[{"f1", "[", "t", "]"}], ",", " ", 
      RowBox[{
       RowBox[{
        RowBox[{"(", "if_InterpolatingFunction", ")"}], "[", "t", "]"}], 
       "\[Rule]", 
       RowBox[{"InterpolatingFunctionCoordinates", "[", "if", "]"}]}], ",", 
      " ", "Infinity"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"1", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"pode", " ", "=", " ", 
  RowBox[{"Show", "[", 
   RowBox[{
    RowBox[{"Block", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"$DisplayFunction", " ", "=", " ", "Identity"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"ListPlot", "[", 
         RowBox[{
          RowBox[{"Transpose", "[", 
           RowBox[{"{", 
            RowBox[{"t1", ",", " ", 
             RowBox[{
              RowBox[{
               SubscriptBox["f", "exact"], "[", "t1", "]"}], " ", "-", " ", 
              RowBox[{"f1", "[", "t1", "]"}]}]}], "}"}], "]"}], ",", " ", 
          RowBox[{"PlotStyle", "\[Rule]", 
           RowBox[{"PointSize", "[", ".02", "]"}]}]}], "]"}], ",", 
        RowBox[{"Plot", "[", 
         RowBox[{
          RowBox[{
           RowBox[{
            SubscriptBox["f", "exact"], "[", "t", "]"}], "-", " ", 
           RowBox[{"f1", "[", "t", "]"}]}], ",", 
          RowBox[{"{", 
           RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", " ", 
          RowBox[{"PlotStyle", "\[Rule]", 
           RowBox[{"RGBColor", "[", 
            RowBox[{".8", ",", ".8", ",", ".8"}], "]"}]}]}], "]"}], ",", 
        RowBox[{"Plot", "[", 
         RowBox[{
          RowBox[{
           RowBox[{
            SubscriptBox["f", "exact"], "[", "t", "]"}], "-", " ", 
           RowBox[{
            SubscriptBox["f1", "dense"], "[", "t", "]"}]}], ",", 
          RowBox[{"{", 
           RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "}"}]}], 
     "]"}], ",", " ", 
    RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}]}], "Input",
 CellLabel->"In[73]:=",
 CellID->1698780919],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 197},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[74]=",
 CellID->361450972]
}, Open  ]],

Cell["\<\
From the plot, it is quite apparent that when the time steps get large, the \
default solution output has much larger error between time steps. The dense \
output solution represents the solution computed by the solver even between \
time steps. Keep in mind, however, that the dense output solution takes up \
much more space:\
\>", "Text",
 CellID->1624873348],

Cell["\<\
This compares the sizes of the default and dense output solutions.\
\>", "MathCaption",
 CellID->1202072407],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ByteCount", " ", "/@", " ", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"f1", "[", "t", "]"}], ",", " ", 
    RowBox[{
     SubscriptBox["f1", "dense"], "[", "t", "]"}]}], "}"}]}]], "Input",
 CellLabel->"In[75]:=",
 CellID->639802234],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"3560", ",", "17648"}], "}"}]], "Output",
 ImageSize->{95, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[75]=",
 CellID->156346419]
}, Open  ]],

Cell["\<\
When the quantity you want to derive from the solution is complicated, you \
can ensure that it is locally kept within tolerances by giving it as an \
algebraic quantity, forcing the solver to keep its error in control.\
\>", "Text",
 CellID->925301436],

Cell["\<\
This adds a dependent variable with an algebraic equation that sets the \
dependent variable equal to the quantity of interest.\
\>", "MathCaption",
 CellID->23158045],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"f2", "[", "t_", "]"}], " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{
    RowBox[{"g", "[", "t", "]"}], " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"eqns", ",", " ", "ics", ",", " ", 
        RowBox[{
         RowBox[{"g", "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"f", "[", "t", "]"}]}]}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y", ",", "g"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "]"}]}]], "Input",\

 CellLabel->"In[76]:=",
 CellID->1368630866],

Cell[BoxData[
 RowBox[{
  TagBox[
   RowBox[{"InterpolatingFunction", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
   False,
   Editable->False], "[", "t", "]"}]], "Output",
 ImageSize->{279, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[76]=",
 CellID->514741714]
}, Open  ]],

Cell["This computes the same solution with dense output", "MathCaption",
 CellID->349454940],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   SubscriptBox["f2", "dense"], "[", "t_", "]"}], " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{
    RowBox[{"g", "[", "t", "]"}], " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"eqns", ",", " ", "ics", ",", " ", 
        RowBox[{
         RowBox[{"g", "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{"f", "[", "t", "]"}]}]}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"x", ",", "y", ",", "g"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", " ", 
      RowBox[{"InterpolationOrder", "\[Rule]", "All"}]}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[77]:=",
 CellID->1060112390],

Cell[BoxData[
 RowBox[{
  TagBox[
   RowBox[{"InterpolatingFunction", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "1.`"}], "}"}], "}"}], ",", "\<\"<>\"\>"}], "]"}],
   False,
   Editable->False], "[", "t", "]"}]], "Output",
 ImageSize->{279, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[77]=",
 CellID->661591095]
}, Open  ]],

Cell["\<\
This makes a plot comparing the error for all four solutions. The time steps \
for IDA are shown as blue points and the dense output from IDA is shown in \
blue with the default output shown in light blue.\
\>", "MathCaption",
 CellID->337772775],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"t2", " ", "=", 
   RowBox[{
    RowBox[{"InterpolatingFunctionCoordinates", "[", 
     RowBox[{"Head", "[", 
      RowBox[{"f2", "[", "t", "]"}], "]"}], "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"pode", ",", " ", 
     RowBox[{"ListPlot", "[", 
      RowBox[{
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"t2", ",", " ", 
          RowBox[{
           RowBox[{
            SubscriptBox["f", "exact"], "[", "t2", "]"}], " ", "-", " ", 
           RowBox[{"f2", "[", "t2", "]"}]}]}], "}"}], "]"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"RGBColor", "[", 
           RowBox[{"0", ",", "0", ",", "1"}], "]"}], ",", 
          RowBox[{"PointSize", "[", "0.02", "]"}]}], "}"}]}]}], "]"}], ",", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["f", "exact"], "[", "t", "]"}], "-", " ", 
        RowBox[{"f2", "[", "t", "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"RGBColor", "[", 
         RowBox[{".7", ",", ".7", ",", "1"}], "]"}]}]}], "]"}], ",", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{
        RowBox[{
         SubscriptBox["f", "exact"], "[", "t", "]"}], "-", " ", 
        RowBox[{
         SubscriptBox["f2", "dense"], "[", "t", "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", " ", 
       RowBox[{"PlotStyle", "\[Rule]", 
        RowBox[{"RGBColor", "[", 
         RowBox[{"0", ",", "0", ",", "1"}], "]"}]}]}], "]"}]}], "}"}], ",", 
   " ", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}], ",", " ", 
      RowBox[{"1*^-7", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"-", "1"}], ",", "1"}], "}"}]}]}], "}"}]}]}], 
  "]"}]}], "Input",
 CellLabel->"In[78]:=",
 CellID->1615742369],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 203},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[79]=",
 CellID->84264753]
}, Open  ]],

Cell["\<\
You can see from the plot that the error is somewhat smaller when the \
quantity is computed algebraically along with the solution.\
\>", "Text",
 CellID->478568206],

Cell[TextData[{
 "The remainder of this documentation will focus on suboptions of the two \
possible settings for the ",
 StyleBox["ImplicitSolver", "MR"],
 " option, which can be ",
 StyleBox["Newton", "MR"],
 " or ",
 StyleBox["GMRES", "MR"],
 ". With ",
 StyleBox["Newton", "MR"],
 ", the Jacobian or an approximation to it is computed and the linear system \
is solved to find the Newton step. On the other hand, ",
 StyleBox["GMRES", "MR"],
 " is an iterative method and, rather than computing the entire Jacobian, a \
directional derivative is computed for each iterative step."
}], "Text",
 CellID->1027181813],

Cell[TextData[{
 "The ",
 StyleBox["Newton", "MR"],
 " method has one method option, ",
 StyleBox["LinearSolveMethod", "MR"],
 ", which you can use to tell ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " how to solve the linear equations required to find the Newton step. There \
are several possible values for this option."
}], "Text",
 CellID->1505708498],

Cell[BoxData[GridBox[{
   {
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "this is the default. Automatically switch between using the ",
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " ",
     Cell[BoxData[
      ButtonBox["LinearSolve",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/LinearSolve"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      ButtonBox["Band",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Band"]], "InlineFormula"],
     " methods depending on the band width of the Jacobian. For systems with \
larger band width, this will automatically switch to a direct sparse solver \
for large systems with sparse Jacobians "
    }], "TableText"]},
   {"Band", Cell["\<\
use the IDA band method (see the IDA user manual for more information)\
\>", "TableText"]},
   {Cell["Dense"], Cell["\<\
use the IDA dense method (see the IDA user manual for more information) \
\>", "TableText"]}
  }]], "DefinitionBox",
 CellID->6899675],

Cell[TextData[{
 "Possible settings for the ",
 Cell[BoxData["LinearSolveMethod"], "InlineFormula"],
 " option. "
}], "Caption",
 CellID->1342859599],

Cell[TextData[{
 "The ",
 StyleBox["GMRES", "MR"],
 " method may be substantially faster, but is typically quite a bit more \
tricky to use because to really be effective typically requires a \
preconditioner, which can be supplied via a method option. There are also \
some other method options that control the Krylov subspace process. To use \
these, refer to the IDA user guide [",
 ButtonBox["HT99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#17230"],
 "]."
}], "Text",
 CellID->385809085],

Cell[BoxData[GridBox[{
   {Cell["GMRES method option name", "TableHeader"], Cell[
    "default value", "TableHeader"], " "},
   {"Preconditioner", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "a ",
     StyleBox["Mathematica",
      FontSlant->"Italic"],
     " function that returns another function that preconditions"
    }], "TableText"]},
   {"OrthogonalizationType", "ModifiedGramSchmidt", Cell[TextData[{
     "this can also be ",
     Cell[BoxData["ClassicalGramSchmidt"], "InlineFormula"],
     StyleBox[" (s", "InlineCode"],
     "ee variable ",
     Cell[BoxData["gstype"], "InlineFormula"],
     " in the IDA User guide)"
    }], "TableText"]},
   {"MaxKrylovSubspaceDimension", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "maximum susbspace dimension (see variable ",
     Cell[BoxData["maxl"], "InlineFormula"],
     " in the IDA user guide)"
    }], "TableText"]},
   {"MaxKrylovRestarts", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "maximum number of restarts (see variable ",
     Cell[BoxData["maxrs"], "InlineFormula"],
     " in the IDA user guide)"
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->258276674],

Cell[TextData[{
 Cell[BoxData["GMRES"], "InlineFormula"],
 " method options. "
}], "Caption",
 CellID->293348272],

Cell["\<\
As an example problem, consider a two-dimensional Burgers\[CloseCurlyQuote] \
equation. \
\>", "Text",
 CellID->1564531112],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["u", "t"], "=", 
   RowBox[{
    RowBox[{"\[Nu]", " ", 
     RowBox[{"(", 
      RowBox[{
       SubscriptBox["u", "xx"], "+", 
       SubscriptBox["u", "yy"]}], ")"}]}], "-", 
    RowBox[{
     FractionBox["1", "2"], 
     RowBox[{"(", 
      RowBox[{
       SubscriptBox[
        RowBox[{"(", 
         SuperscriptBox["u", "2"], ")"}], "x"], "+", 
       SubscriptBox[
        RowBox[{"(", 
         SuperscriptBox["u", "2"], ")"}], "y"]}], ")"}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->1235291425],

Cell[TextData[{
 "This can typically be solved with an ordinary differential equation solver, \
but in this example two things are achieved by using the DAE solver. First, \
boundary conditions are enforced as algebraic conditions. Second, ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " is forced to use conservative differencing by using an algebraic term. For \
comparison, a known exact solution will be used for initial and boundary \
conditions."
}], "Text",
 CellID->527741282],

Cell["\<\
This defines a function that satisfies Burger\[CloseCurlyQuote]s equation.\
\>", "MathCaption",
 CellID->1139913670],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Bsol", "[", 
    RowBox[{"t_", ",", " ", "x_", ",", " ", "y_"}], "]"}], " ", "=", " ", 
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{"Exp", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"x", "+", "y", "-", "t"}], ")"}], "/", 
        RowBox[{"(", 
         RowBox[{"2", " ", "\[Nu]"}], ")"}]}], "]"}]}], ")"}]}]}], 
  ";"}]], "Input",
 CellLabel->"In[80]:=",
 CellID->1935407110],

Cell["\<\
This defines initial and boundary conditions for the unit square consistent \
with the exact solution.\
\>", "MathCaption",
 CellID->1108422822],

Cell[BoxData[{
 RowBox[{
  RowBox[{"ic", " ", "=", " ", 
   RowBox[{
    RowBox[{"u", "[", 
     RowBox[{"0", ",", "x", ",", "y"}], "]"}], " ", "\[Equal]", " ", 
    RowBox[{"Bsol", "[", 
     RowBox[{"0", ",", "x", ",", "y"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"bc", " ", "=", " ", 
   RowBox[{"{", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "0", ",", "y"}], "]"}], " ", "\[Equal]", " ", 
      RowBox[{"Bsol", "[", 
       RowBox[{"t", ",", "0", ",", "y"}], "]"}]}], ",", " ", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "1", ",", "y"}], "]"}], " ", "\[Equal]", " ", 
      RowBox[{"Bsol", "[", 
       RowBox[{"t", ",", "1", ",", "y"}], "]"}]}], ",", "\[IndentingNewLine]",
      
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x", ",", "0"}], "]"}], " ", "\[Equal]", " ", 
      RowBox[{"Bsol", "[", 
       RowBox[{"t", ",", "x", ",", "0"}], "]"}]}], ",", " ", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x", ",", "1"}], "]"}], " ", "\[Equal]", " ", 
      RowBox[{"Bsol", "[", 
       RowBox[{"t", ",", "x", ",", "1"}], "]"}]}]}], "}"}]}], ";"}]}], "Input",\

 CellLabel->"In[81]:=",
 CellID->1771501574],

Cell["This defines the differential equation.", "MathCaption",
 CellID->1486704381],

Cell[BoxData[
 RowBox[{
  RowBox[{"de", " ", "=", " ", 
   RowBox[{
    RowBox[{"D", "[", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "t"}], "]"}], " ", 
    "\[Equal]", " ", 
    RowBox[{
     RowBox[{"\[Nu]", 
      RowBox[{"(", " ", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "x", ",", "x"}], 
         "]"}], " ", "+", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "y", ",", "y"}], 
         "]"}]}], ")"}]}], " ", "-", 
     RowBox[{
      RowBox[{"u", "[", 
       RowBox[{"t", ",", "x", ",", "y"}], "]"}], 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "x"}], "]"}], " ", 
        "+", " ", 
        RowBox[{"D", "[", 
         RowBox[{
          RowBox[{"u", "[", 
           RowBox[{"t", ",", "x", ",", "y"}], "]"}], ",", "y"}], "]"}]}], 
       ")"}]}]}]}]}], ";"}]], "Input",
 CellLabel->"In[83]:=",
 CellID->618888538],

Cell[TextData[{
 "This sets the diffusion constant \[Nu] to a value for which we can find a \
solution in a reasonable amount of time and shows a plot of the solution at \
",
 StyleBox["t",
  FontSlant->"Italic"],
 " == 1."
}], "MathCaption",
 CellID->1248891274],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"\[Nu]", " ", "=", " ", "0.025"}], ";"}], "\n", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Bsol", "[", 
    RowBox[{"1", ",", "x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[84]:=",
 CellID->541293140],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 289},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[85]=",
 CellID->341297289]
}, Open  ]],

Cell["\<\
You can see from the plot that with \[Nu] = 0.025, there is a fairly steep \
front. This moves with constant speed.\
\>", "Text",
 CellID->391786532],

Cell[TextData[{
 "This solves the problem using the default settings for ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " and the IDA method with the exception of the ",
 StyleBox["DifferentiateBoundaryConditions", "MR"],
 " option for ",
 StyleBox["MethodOfLines", "MR"],
 ", which causes ",
 StyleBox["NDSolve", "MR"],
 " to treat the boundary conditions as algebraic. "
}], "MathCaption",
 CellID->1466594576],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", "[", 
  RowBox[{"sol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"de", ",", " ", "ic", ",", " ", "bc"}], "}"}], ",", "u", ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", " ", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
        RowBox[{
        "\"\<DifferentiateBoundaryConditions\>\"", "\[Rule]", "False"}]}], 
       "}"}]}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[86]:=",
 CellID->1281848817],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.2330000000000005`", ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"u", "\[Rule]", 
      TagBox[
       RowBox[{"InterpolatingFunction", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0.`", ",", "1.`"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0.`", ",", "1.`"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0.`", ",", "1.`"}], "}"}]}], "}"}], ",", "\<\"<>\"\>"}], 
        "]"}],
       False,
       Editable->False]}], "}"}], "}"}]}], "}"}]], "Output",
 ImageSize->{498, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[86]=",
 CellID->490354485]
}, Open  ]],

Cell["\<\
Since there is an exact solution to compare to, the overall solution accuracy \
can be compared as well.\
\>", "Text",
 CellID->459651062],

Cell["\<\
This defines a function that finds the maximum deviation between the exact \
and computed solutions at the grid points at all of the time steps.\
\>", "MathCaption",
 CellID->1631303758],

Cell[BoxData[
 RowBox[{
  RowBox[{"errfun", "[", "sol_", "]"}], " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"ifun", " ", "=", " ", 
       RowBox[{"First", "[", 
        RowBox[{"u", " ", "/.", " ", "sol"}], "]"}]}], ",", " ", "grid", ",", 
      " ", "exvals", ",", " ", "gvals"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"grid", " ", "=", " ", 
      RowBox[{"InterpolatingFunctionGrid", "[", "ifun", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"gvals", " ", "=", " ", 
      RowBox[{"InterpolatingFunctionValuesOnGrid", "[", "ifun", "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"exvals", " ", "=", " ", 
      RowBox[{"Apply", "[", 
       RowBox[{"Bsol", ",", 
        RowBox[{"Transpose", "[", 
         RowBox[{"grid", ",", " ", 
          RowBox[{"RotateLeft", "[", 
           RowBox[{
            RowBox[{"Range", "[", 
             RowBox[{"ArrayDepth", "[", "grid", "]"}], "]"}], ",", "1"}], 
           "]"}]}], "]"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"Max", "[", 
      RowBox[{"Abs", "[", 
       RowBox[{"exvals", " ", "-", " ", "gvals"}], "]"}], "]"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[87]:=",
 CellID->673046140],

Cell["This computes the maximal error for the computed solution. ", \
"MathCaption",
 CellID->146939956],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"errfun", "[", "sol", "]"}]], "Input",
 CellLabel->"In[88]:=",
 CellID->1969076533],

Cell[BoxData["0.000749446242192614`"], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[88]=",
 CellID->3027874]
}, Open  ]],

Cell["\<\
In the following, a comparison will be made with different settings for the \
options of the IDA method. To emphasize the option settings, a function will \
be defined to time the computation of the solution and give the maximal \
error.\
\>", "Text",
 CellID->2103340506],

Cell["\<\
This defines a function for comparing different IDA option settings.\
\>", "MathCaption",
 CellID->300161092],

Cell[BoxData[
 RowBox[{
  RowBox[{"TimeSolution", "[", "idaopts___", "]"}], " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"time", ",", " ", "err", ",", " ", "steps"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"time", " ", "=", " ", 
      RowBox[{"First", "[", 
       RowBox[{"Timing", "[", 
        RowBox[{"sol", " ", "=", " ", 
         RowBox[{"NDSolve", "[", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"de", ",", " ", "ic", ",", " ", "bc"}], "}"}], ",", "u", 
           ",", 
           RowBox[{"{", 
            RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", " ", 
           RowBox[{"Method", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"\"\<MethodOfLines\>\"", ",", " ", 
              RowBox[{
              "\"\<DifferentiateBoundaryConditions\>\"", "\[Rule]", "False"}],
               ",", " ", 
              RowBox[{"Method", "\[Rule]", 
               RowBox[{"{", 
                RowBox[{"IDA", ",", " ", "idaopts"}], "}"}]}]}], "}"}]}]}], 
          "]"}]}], "]"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"err", " ", "=", " ", 
      RowBox[{"errfun", "[", "sol", "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"steps", " ", "=", " ", 
      RowBox[{
       RowBox[{"Length", "[", 
        RowBox[{"First", "[", 
         RowBox[{"InterpolatingFunctionCoordinates", "[", 
          RowBox[{"First", "[", 
           RowBox[{"u", " ", "/.", " ", "sol"}], "]"}], "]"}], "]"}], "]"}], 
       " ", "\"\<Steps\>\""}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"time", ",", " ", "err", ",", " ", "steps"}], "}"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[89]:=",
 CellID->1168486674],

Cell["No options use the previous defaults.", "MathCaption",
 CellID->1461565357],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TimeSolution", "[", "]"}]], "Input",
 CellLabel->"In[90]:=",
 CellID->307060767],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.184`", ",", "0.000749446242192614`", ",", 
   RowBox[{"88", " ", "\<\"Steps\"\>"}]}], "}"}]], "Output",
 ImageSize->{205, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[90]=",
 CellID->95766552]
}, Open  ]],

Cell[TextData[{
 "This uses the ",
 StyleBox["Band", "MR"],
 " method."
}], "MathCaption",
 CellID->1847374442],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TimeSolution", "[", 
  RowBox[{"\"\<ImplicitSolver\>\"", "\[Rule]", 
   RowBox[{"{", 
    RowBox[{"\"\<Newton\>\"", ",", " ", 
     RowBox[{"\"\<LinearSolveMethod\>\"", "\[Rule]", "\"\<Band\>\""}]}], 
    "}"}]}], "]"}]], "Input",
 CellLabel->"In[91]:=",
 CellID->565337442],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"8.543`", ",", "0.0007494967792602281`", ",", 
   RowBox[{"88", " ", "\<\"Steps\"\>"}]}], "}"}]], "Output",
 ImageSize->{205, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[91]=",
 CellID->83512217]
}, Open  ]],

Cell[TextData[{
 "The ",
 StyleBox["Band", "MR"],
 " method is not very effective because the bandwidth of the Jacobian is \
relatively large, partly because of the fourth order derivatives used and \
partly because the one-sided stencils used near the boundary add width at the \
top and bottom. You can specify the bandwidth explicitly."
}], "Text",
 CellID->1264421906],

Cell[TextData[{
 "This uses the ",
 StyleBox["Band", "MR"],
 " method with the width set to include the stencil of the differences in \
only one direction."
}], "MathCaption",
 CellID->1800628867],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TimeSolution", "[", 
  RowBox[{"\"\<ImplicitSolver\>\"", "\[Rule]", 
   RowBox[{"{", 
    RowBox[{"\"\<Newton\>\"", ",", " ", 
     RowBox[{"\"\<LinearSolveMethod\>\"", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<Band\>\"", ",", " ", 
        RowBox[{"\"\<BandWidth\>\"", "\[Rule]", "3"}]}], "}"}]}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[92]:=",
 CellID->1184952180],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"7.440999999999999`", ",", "0.0009379619212770418`", ",", 
   RowBox[{"311", " ", "\<\"Steps\"\>"}]}], "}"}]], "Output",
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[92]=",
 CellID->456262799]
}, Open  ]],

Cell["\<\
While the solution time was smaller, notice that the error is slightly \
greater and the total number of time steps is a lot greater. If the problem \
was more stiff, the iterations likely would not have converged because it was \
missing information from the other direction. Ideally, the bandwidth should \
not eliminate information from an entire dimension. \
\>", "Text",
 CellID->1588552329],

Cell[TextData[{
 "This computes the grids used in the ",
 StyleBox["X",
  FontSlant->"Italic"],
 " and ",
 StyleBox["Y",
  FontSlant->"Italic"],
 " directions and shows the number of points used."
}], "MathCaption",
 CellID->145891354],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"X", ",", "Y"}], "}"}], " ", "=", " ", 
   RowBox[{
    RowBox[{"InterpolatingFunctionCoordinates", "[", 
     RowBox[{"First", "[", 
      RowBox[{"u", " ", "/.", " ", "sol"}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"{", 
      RowBox[{"2", ",", "3"}], "}"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"{", 
   RowBox[{"nx", ",", "ny"}], "}"}], " ", "=", " ", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Length", "[", "X", "]"}], ",", " ", 
    RowBox[{"Length", "[", "Y", "]"}]}], "}"}]}]}], "Input",
 CellLabel->"In[93]:=",
 CellID->2028353503],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"51", ",", "51"}], "}"}]], "Output",
 ImageSize->{58, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[94]=",
 CellID->149060360]
}, Open  ]],

Cell[TextData[{
 "This uses the ",
 StyleBox["Band", "MR"],
 " method with the width set to include at least part of the stencil in both \
directions."
}], "MathCaption",
 CellID->764779892],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TimeSolution", "[", 
  RowBox[{"\"\<ImplicitSolver\>\"", "\[Rule]", 
   RowBox[{"{", 
    RowBox[{"\"\<Newton\>\"", ",", " ", 
     RowBox[{"\"\<LinearSolveMethod\>\"", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<Band\>\"", ",", " ", 
        RowBox[{"\"\<BandWidth\>\"", "\[Rule]", "51"}]}], "}"}]}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[95]:=",
 CellID->1129761129],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.2730000000000006`", ",", "0.0008597296989719672`", ",", 
   RowBox[{"88", " ", "\<\"Steps\"\>"}]}], "}"}]], "Output",
 ImageSize->{198, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[95]=",
 CellID->245304967]
}, Open  ]],

Cell[TextData[{
 "With the more appropriate setting of the bandwidth, the solution is still \
slightly slower than in the default case. The ",
 StyleBox["Band", "MR"],
 " method can sometimes be effective on two-dimensional problems, but is \
usually most effective on one-dimensional problems."
}], "Text",
 CellID->2023470664],

Cell[TextData[{
 "This computes the solution using the ",
 StyleBox["GMRES", "MR"],
 " implicit solver without a preconditioner."
}], "MathCaption",
 CellID->1521547661],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TimeSolution", "[", 
  RowBox[{"\"\<ImplicitSolver\>\"", "\[Rule]", "\"\<GMRES\>\""}], 
  "]"}]], "Input",
 CellLabel->"In[96]:=",
 CellID->788177130],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"26.137`", ",", "0.004354313347916278`", ",", 
   RowBox[{"672", " ", "\<\"Steps\"\>"}]}], "}"}]], "Output",
 ImageSize->{212, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[96]=",
 CellID->2385253]
}, Open  ]],

Cell[TextData[{
 "This is incredibly slow! Using the ",
 StyleBox["GMRES", "MR"],
 " method without a preconditioner is not recommended for this very reason. \
However, finding a good preconditioner is not usually trivial. For this \
example, a diagonal preconditioner will be used."
}], "Text",
 CellID->1134481412],

Cell[TextData[{
 "The setting of the ",
 StyleBox["Preconditioner", "MR"],
 " option should be a function ",
 StyleBox["f", "MR"],
 ", which accepts four arguments that will be given to it by ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " such that ",
 StyleBox["f[t, x, x', c]", "MR"],
 " returns another function that will apply the preconditioner to the \
residual vector. (See IDA user guide [",
 ButtonBox["HT99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#17230"],
 "] for details on how the preconditioner is used.) The arguments ",
 StyleBox["t, x, x', c", "MR"],
 " are the current time, solution vector, solution derivative vector, and the \
constant ",
 StyleBox["c",
  FontSlant->"Italic"],
 " in formula (2) above. For example, if you can determine a procedure that \
would generate an appropriate preconditioner matrix ",
 StyleBox["P",
  FontSlant->"Italic"],
 " as a function of these arguments, you could use\n",
 StyleBox["Preconditioner->Function[{t,x,xp,c}, LinearSolve[P[t,x,xp,c]]]", 
  "MR"],
 "\nto produce a ",
 Cell[BoxData[
  ButtonBox["LinearSolveFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/LinearSolveFunction"]], "InlineFormula"],
 " object which will effectively invert the preconditioner matrix ",
 StyleBox["P",
  FontSlant->"Italic"],
 ". Typically, for each time the preconditioner function is set up, it is \
applied to the residual vector several times, so using some sort of \
factorization such as is contained in a ",
 StyleBox["LinearSolveFunction", "MR"],
 " is a good idea."
}], "Text",
 CellID->1247828192],

Cell["\<\
For the diagonal case, the inverse can be effected simply by using the \
reciprocal. The most difficult part of setting up a diagonal preconditioner \
is keeping in mind that values on the boundary should not be affected by it. \
\
\>", "Text",
 CellID->641655944],

Cell["\<\
This finds the diagonal elements of the differentiation matrix for computing \
the preconditioner.\
\>", "MathCaption",
 CellID->1807527552],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"DM", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"2", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"X", ",", "Y"}], "}"}]}], "]"}], "@", 
     "\"\<DifferentiationMatrix\>\""}], " ", "+", " ", 
    RowBox[{
     RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"X", ",", "Y"}], "}"}]}], "]"}], "@", 
     "\"\<DifferentiationMatrix\>\""}]}]}], ";"}], "\n", 
 RowBox[{"Short", "[", 
  RowBox[{"diag", " ", "=", " ", 
   RowBox[{"Tr", "[", 
    RowBox[{"DM", ",", "List"}], "]"}]}], "]"}]}], "Input",
 CellLabel->"In[97]:=",
 CellID->1394768149],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{"18750.`", ",", "6250.`", ",", "3125.`", ",", "3125.`", ",", 
    RowBox[{"\[LeftSkeleton]", "2593", "\[RightSkeleton]"}], ",", "3125.`", 
    ",", "3125.`", ",", "6250.000000000001`", ",", "18750.`"}], "}"}],
  Short]], "Output",
 ImageSize->{462, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[98]//Short=",
 CellID->56429766]
}, Open  ]],

Cell["\<\
This gets the positions where elements at the boundary that satisfy a simple \
algebraic condition are in the flattened solution vector.\
\>", "MathCaption",
 CellID->1948832414],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"bound", " ", "=", " ", 
   RowBox[{"SparseArray", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", " ", "1"}], "}"}], "\[Rule]", "1."}], ",", " ", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i_", ",", " ", "ny"}], "}"}], "\[Rule]", "1."}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1", ",", "i_"}], "}"}], "\[Rule]", "1."}], ",", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"nx", ",", "i_"}], "}"}], "\[Rule]", "1."}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"nx", ",", "ny"}], "}"}], ",", "0."}], "]"}]}], ";"}], "\n", 
 RowBox[{"Short", "[", 
  RowBox[{"pos", " ", "=", " ", 
   RowBox[{
    RowBox[{"Drop", "[", 
     RowBox[{
      RowBox[{"ArrayRules", "[", 
       RowBox[{"Flatten", "[", "bound", "]"}], "]"}], ",", 
      RowBox[{"-", "1"}]}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"All", ",", "1", ",", "1"}], "]"}], "]"}]}], "]"}]}], "Input",
 CellLabel->"In[99]:=",
 CellID->1350625007],

Cell[BoxData[
 TagBox[
  RowBox[{"{", 
   RowBox[{
   "1", ",", "2", ",", "3", ",", "4", ",", "5", ",", "6", ",", "7", ",", "8", 
    ",", "9", ",", "10", ",", 
    RowBox[{"\[LeftSkeleton]", "180", "\[RightSkeleton]"}], ",", "2592", ",", 
    "2593", ",", "2594", ",", "2595", ",", "2596", ",", "2597", ",", "2598", 
    ",", "2599", ",", "2600", ",", "2601"}], "}"}],
  Short]], "Output",
 ImageSize->{360, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[100]//Short=",
 CellID->690628779]
}, Open  ]],

Cell["\<\
This defines the function that sets up the function called to get the \
effective inverse of the preconditioner. For the diagonal case, the inverse \
is done simply by taking the reciprocal.\
\>", "MathCaption",
 CellID->969835440],

Cell[BoxData[
 RowBox[{
  RowBox[{"pfun", "[", 
   RowBox[{"t_", ",", " ", "x_", ",", " ", "xp_", ",", " ", "c_"}], "]"}], 
  " ", ":=", " ", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"d", ",", "dd"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"d", " ", "=", " ", 
      RowBox[{"1.", "/", 
       RowBox[{"(", 
        RowBox[{"c", " ", "-", " ", 
         RowBox[{"\[Nu]", " ", "diag"}]}], ")"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"d", "[", 
       RowBox[{"[", "pos", "]"}], "]"}], " ", "=", " ", "1."}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"Function", "[", 
       RowBox[{"#", " ", "dd"}], "]"}], " ", "/.", " ", 
      RowBox[{"dd", "\[Rule]", "d"}]}]}]}], "]"}]}]], "Input",
 CellLabel->"In[101]:=",
 CellID->664706716],

Cell[TextData[{
 "This uses the preconditioned ",
 StyleBox["GMRES", "MR"],
 " method to compute the solution."
}], "MathCaption",
 CellID->1070470689],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TimeSolution", "[", 
  RowBox[{"\"\<ImplicitSolver\>\"", "\[Rule]", 
   RowBox[{"{", 
    RowBox[{"\"\<GMRES\>\"", ",", " ", 
     RowBox[{"\"\<Preconditioner\>\"", "\[Rule]", "pfun"}]}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[102]:=",
 CellID->1333698066],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.1609999999999965`", ",", "0.0007160064681386302`", ",", 
   RowBox[{"88", " ", "\<\"Steps\"\>"}]}], "}"}]], "Output",
 ImageSize->{205, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[102]=",
 CellID->57846478]
}, Open  ]],

Cell[TextData[{
 "Thus, even with a crude preconditioner, the ",
 StyleBox["GMRES", "MR"],
 " method computes the solution faster than the using the direct sparse \
solvers. "
}], "Text",
 CellID->1635808131],

Cell[TextData[{
 "For PDE discretizations with higher-order temporal derivatives or systems \
of PDEs, you may need to look at the corresponding ",
 StyleBox[ButtonBox["NDSolve`StateData",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveStateData"], "MR"],
 " object to determine how the variables are ordered so that you can get the \
structural form of the preconditioner correctly."
}], "Text",
 CellID->1990908964]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"IDA Method for NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/NDSolveIntroductoryTutorialDAEs"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/NDSolveIntroductoryTutorialStateData#2125957807"]}],
       "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 50.1807416}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "The IDA package is part of the SUNDIALS (SUite of Nonlinear and \
DIfferential/ALgebraic equation Solvers) developed at the Center for Applied \
Scientific Computing of Lawrence Livermore National Laboratory. As described \
in the IDA user guide [HT99], 'IDA is a general purpose solver for the \
initial value problem for systems of differential-algebraic equations (DAEs). \
The name IDA stands for Implicit Differential-Algebraic solver. IDA is based \
on DASPK ...' DASPK [BHP94], [BHP98] is a FORTRAN code for solving large \
scale differential-algebraic systems. In Mathematica, an interface has been \
provided to the IDA package so that rather than needing to write a function \
in C for evaluating the residual and compiling the program, Mathematica \
generates the function automatically from the equations you input to NDSolve. \
IDA solves the system (1) with Backward Differentiation Formula (BDF) methods \
of orders 1 through 5, implemented using a variable-step form. The BDF of \
order k is at time t_n= t_n-1 + h_n is given by the formula", 
    "synonyms" -> {}, "title" -> "IDA Method for NDSolve", "type" -> 
    "Tutorial", "uri" -> "tutorial/NDSolveIDAMethod"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:3"->{
  Cell[1367, 43, 111, 2, 70, "Title",
   CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"},
   CellID->1283904336]},
 "b:11.0"->{
  Cell[1367, 43, 111, 2, 70, "Title",
   CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"},
   CellID->1283904336]},
 "ndsg:4.1.0"->{
  Cell[1367, 43, 111, 2, 70, "Title",
   CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"},
   CellID->1283904336]},
 "IDA"->{
  Cell[1367, 43, 111, 2, 70, "Title",
   CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"},
   CellID->1283904336]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:3", 114699, 2909},
 {"b:11.0", 114828, 2913},
 {"ndsg:4.1.0", 114961, 2917},
 {"IDA", 115087, 2921}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 742, 16, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1367, 43, 111, 2, 70, "Title",
 CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"},
 CellID->1283904336],
Cell[1481, 47, 1220, 33, 70, "Text",
 CellID->544144180],
Cell[2704, 82, 545, 17, 70, "Text",
 CellID->1870584582],
Cell[3252, 101, 513, 17, 70, "Text",
 CellID->346288529],
Cell[3768, 120, 483, 17, 70, "DisplayMath",
 CellID->1056205345],
Cell[4254, 139, 354, 12, 70, "Text",
 CellID->998665234],
Cell[4611, 153, 554, 19, 70, "DisplayMath",
 CellID->1077267824],
Cell[5168, 174, 157, 4, 70, "Text",
 CellID->993285444],
Cell[5328, 180, 656, 23, 70, "NumberedEquation",
 CellID->1650775136],
Cell[5987, 205, 805, 22, 70, "Text",
 CellID->1129336518],
Cell[6795, 229, 956, 38, 70, "Text",
 CellID->1812989206],
Cell[7754, 269, 484, 16, 70, "DisplayMath",
 CellID->1870712677],
Cell[8241, 287, 382, 16, 70, "Text",
 CellID->1530839459],
Cell[8626, 305, 551, 16, 70, "Text",
 CellID->78251959],
Cell[9180, 323, 226, 9, 70, "MathCaption",
 CellID->215238697],
Cell[CellGroupData[{
Cell[9431, 336, 114, 3, 70, "Input",
 CellID->326296895],
Cell[9548, 341, 326, 10, 36, "Output",
 CellID->11683416]
}, Open  ]],
Cell[9889, 354, 558, 12, 70, "DefinitionBox3Col",
 CellID->694232409],
Cell[10450, 368, 111, 4, 70, "Caption",
 CellID->2135801483],
Cell[10564, 374, 1190, 31, 70, "Text",
 CellID->555629979],
Cell[11757, 407, 583, 17, 70, "Text",
 CellID->121628007],
Cell[12343, 426, 156, 4, 70, "MathCaption",
 CellID->140795431],
Cell[12502, 432, 287, 10, 70, "Input",
 CellID->1558923092],
Cell[12792, 444, 116, 3, 70, "MathCaption",
 CellID->1696791596],
Cell[12911, 449, 837, 28, 70, "Input",
 CellID->1675815117],
Cell[13751, 479, 171, 6, 70, "MathCaption",
 CellID->315677786],
Cell[CellGroupData[{
Cell[13947, 489, 440, 15, 70, "Input",
 CellID->1027584235],
Cell[14390, 506, 984, 36, 51, "Output",
 CellID->139758597]
}, Open  ]],
Cell[15389, 545, 137, 4, 70, "Text",
 CellID->1097972321],
Cell[15529, 551, 187, 4, 70, "MathCaption",
 CellID->1178722026],
Cell[CellGroupData[{
Cell[15741, 559, 494, 16, 70, "Input",
 CellID->977014750],
Cell[16238, 577, 752, 28, 36, "Output",
 CellID->325918764]
}, Open  ]],
Cell[17005, 608, 86, 1, 70, "MathCaption",
 CellID->1722866206],
Cell[CellGroupData[{
Cell[17116, 613, 590, 18, 70, "Input",
 CellID->700115685],
Cell[17709, 633, 752, 28, 36, "Output",
 CellID->483959973]
}, Open  ]],
Cell[18476, 664, 269, 5, 70, "MathCaption",
 CellID->631587903],
Cell[CellGroupData[{
Cell[18770, 673, 2199, 61, 137, "Input",
 CellID->1698780919],
Cell[20972, 736, 10006, 168, 218, "Output",
 Evaluatable->False,
 CellID->361450972]
}, Open  ]],
Cell[30993, 907, 372, 7, 70, "Text",
 CellID->1624873348],
Cell[31368, 916, 118, 3, 70, "MathCaption",
 CellID->1202072407],
Cell[CellGroupData[{
Cell[31511, 923, 255, 8, 70, "Input",
 CellID->639802234],
Cell[31769, 933, 211, 7, 36, "Output",
 CellID->156346419]
}, Open  ]],
Cell[31995, 943, 263, 5, 70, "Text",
 CellID->925301436],
Cell[32261, 950, 177, 4, 70, "MathCaption",
 CellID->23158045],
Cell[CellGroupData[{
Cell[32463, 958, 636, 19, 70, "Input",
 CellID->1368630866],
Cell[33102, 979, 396, 14, 36, "Output",
 CellID->514741714]
}, Open  ]],
Cell[33513, 996, 92, 1, 70, "MathCaption",
 CellID->349454940],
Cell[CellGroupData[{
Cell[33630, 1001, 732, 21, 70, "Input",
 CellID->1060112390],
Cell[34365, 1024, 396, 14, 36, "Output",
 CellID->661591095]
}, Open  ]],
Cell[34776, 1041, 256, 5, 70, "MathCaption",
 CellID->337772775],
Cell[CellGroupData[{
Cell[35057, 1050, 2117, 62, 119, "Input",
 CellID->1615742369],
Cell[37177, 1114, 20509, 340, 224, "Output",
 Evaluatable->False,
 CellID->84264753]
}, Open  ]],
Cell[57701, 1457, 175, 4, 70, "Text",
 CellID->478568206],
Cell[57879, 1463, 617, 16, 70, "Text",
 CellID->1027181813],
Cell[58499, 1481, 365, 11, 70, "Text",
 CellID->1505708498],
Cell[58867, 1494, 1054, 29, 70, "DefinitionBox",
 CellID->6899675],
Cell[59924, 1525, 149, 5, 70, "Caption",
 CellID->1342859599],
Cell[60076, 1532, 521, 13, 70, "Text",
 CellID->385809085],
Cell[60600, 1547, 1529, 41, 70, "DefinitionBox3Col",
 CellID->258276674],
Cell[62132, 1590, 113, 4, 70, "Caption",
 CellID->293348272],
Cell[62248, 1596, 133, 4, 70, "Text",
 CellID->1564531112],
Cell[62384, 1602, 564, 21, 70, "DisplayMath",
 CellID->1235291425],
Cell[62951, 1625, 559, 12, 70, "Text",
 CellID->527741282],
Cell[63513, 1639, 126, 3, 70, "MathCaption",
 CellID->1139913670],
Cell[63642, 1644, 474, 16, 70, "Input",
 CellID->1935407110],
Cell[64119, 1662, 154, 4, 70, "MathCaption",
 CellID->1108422822],
Cell[64276, 1668, 1240, 35, 70, "Input",
 CellID->1771501574],
Cell[65519, 1705, 83, 1, 70, "MathCaption",
 CellID->1486704381],
Cell[65605, 1708, 1195, 39, 70, "Input",
 CellID->618888538],
Cell[66803, 1749, 263, 8, 70, "MathCaption",
 CellID->1248891274],
Cell[CellGroupData[{
Cell[67091, 1761, 392, 12, 47, "Input",
 CellID->541293140],
Cell[67486, 1775, 23507, 389, 310, "Output",
 Evaluatable->False,
 CellID->341297289]
}, Open  ]],
Cell[91008, 2167, 159, 4, 70, "Text",
 CellID->391786532],
Cell[91170, 2173, 486, 14, 70, "MathCaption",
 CellID->1466594576],
Cell[CellGroupData[{
Cell[91681, 2191, 720, 20, 70, "Input",
 CellID->1281848817],
Cell[92404, 2213, 735, 24, 36, "Output",
 CellID->490354485]
}, Open  ]],
Cell[93154, 2240, 148, 4, 70, "Text",
 CellID->459651062],
Cell[93305, 2246, 196, 4, 70, "MathCaption",
 CellID->1631303758],
Cell[93504, 2252, 1263, 33, 70, "Input",
 CellID->673046140],
Cell[94770, 2287, 104, 2, 70, "MathCaption",
 CellID->146939956],
Cell[CellGroupData[{
Cell[94899, 2293, 106, 3, 70, "Input",
 CellID->1969076533],
Cell[95008, 2298, 177, 5, 36, "Output",
 CellID->3027874]
}, Open  ]],
Cell[95200, 2306, 282, 6, 70, "Text",
 CellID->2103340506],
Cell[95485, 2314, 119, 3, 70, "MathCaption",
 CellID->300161092],
Cell[95607, 2319, 1911, 48, 70, "Input",
 CellID->1168486674],
Cell[97521, 2369, 81, 1, 70, "MathCaption",
 CellID->1461565357],
Cell[CellGroupData[{
Cell[97627, 2374, 104, 3, 70, "Input",
 CellID->307060767],
Cell[97734, 2379, 276, 8, 36, "Output",
 CellID->95766552]
}, Open  ]],
Cell[98025, 2390, 111, 5, 70, "MathCaption",
 CellID->1847374442],
Cell[CellGroupData[{
Cell[98161, 2399, 298, 8, 70, "Input",
 CellID->565337442],
Cell[98462, 2409, 277, 8, 36, "Output",
 CellID->83512217]
}, Open  ]],
Cell[98754, 2420, 372, 8, 70, "Text",
 CellID->1264421906],
Cell[99129, 2430, 196, 6, 70, "MathCaption",
 CellID->1800628867],
Cell[CellGroupData[{
Cell[99350, 2440, 408, 11, 70, "Input",
 CellID->1184952180],
Cell[99761, 2453, 291, 8, 36, "Output",
 CellID->456262799]
}, Open  ]],
Cell[100067, 2464, 406, 7, 70, "Text",
 CellID->1588552329],
Cell[100476, 2473, 235, 9, 70, "MathCaption",
 CellID->145891354],
Cell[CellGroupData[{
Cell[100736, 2486, 630, 20, 70, "Input",
 CellID->2028353503],
Cell[101369, 2508, 206, 7, 36, "Output",
 CellID->149060360]
}, Open  ]],
Cell[101590, 2518, 190, 6, 70, "MathCaption",
 CellID->764779892],
Cell[CellGroupData[{
Cell[101805, 2528, 409, 11, 70, "Input",
 CellID->1129761129],
Cell[102217, 2541, 291, 8, 36, "Output",
 CellID->245304967]
}, Open  ]],
Cell[102523, 2552, 328, 7, 70, "Text",
 CellID->2023470664],
Cell[102854, 2561, 169, 5, 70, "MathCaption",
 CellID->1521547661],
Cell[CellGroupData[{
Cell[103048, 2570, 174, 5, 70, "Input",
 CellID->788177130],
Cell[103225, 2577, 277, 8, 36, "Output",
 CellID->2385253]
}, Open  ]],
Cell[103517, 2588, 316, 7, 70, "Text",
 CellID->1134481412],
Cell[103836, 2597, 1669, 44, 70, "Text",
 CellID->1247828192],
Cell[105508, 2643, 274, 6, 70, "Text",
 CellID->641655944],
Cell[105785, 2651, 150, 4, 70, "MathCaption",
 CellID->1807527552],
Cell[CellGroupData[{
Cell[105960, 2659, 824, 25, 70, "Input",
 CellID->1394768149],
Cell[106787, 2686, 418, 11, 50, "Output",
 CellID->56429766]
}, Open  ]],
Cell[107220, 2700, 188, 4, 70, "MathCaption",
 CellID->1948832414],
Cell[CellGroupData[{
Cell[107433, 2708, 1062, 32, 70, "Input",
 CellID->1350625007],
Cell[108498, 2742, 530, 14, 68, "Output",
 CellID->690628779]
}, Open  ]],
Cell[109043, 2759, 241, 5, 70, "MathCaption",
 CellID->969835440],
Cell[109287, 2766, 834, 25, 70, "Input",
 CellID->664706716],
Cell[110124, 2793, 151, 5, 70, "MathCaption",
 CellID->1070470689],
Cell[CellGroupData[{
Cell[110300, 2802, 286, 8, 70, "Input",
 CellID->1333698066],
Cell[110589, 2812, 291, 8, 36, "Output",
 CellID->57846478]
}, Open  ]],
Cell[110895, 2823, 208, 6, 70, "Text",
 CellID->1635808131],
Cell[111106, 2831, 429, 9, 70, "Text",
 CellID->1990908964]
}, Open  ]],
Cell[111550, 2843, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

