(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     52551,       1434]
NotebookOptionsPosition[     45274,       1186]
NotebookOutlinePosition[     48164,       1274]
CellTagsIndexPosition[     47848,       1261]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], " "}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Numerical methods for solving the Lotka-Volterra equations", "Title",
 CellID->1637096269],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->1174568789],

Cell["\<\
The Lotka-Volterra system arises in mathematical biology and models the \
growth of animal species.\
\>", "Text",
 CellID->1832907302],

Cell[TextData[{
 "Consider two species where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", "1"], "(", "T", ")"}], TraditionalForm]]],
 "denotes the number of predators and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["Y", "2"], "(", "T", ")"}], " "}], TraditionalForm]]],
 " denotes the number of prey."
}], "Text",
 CellID->2048574980],

Cell["\<\
A particular case of the Lotka-Volterra differential system is:\
\>", "Text",
 CellID->890047143],

Cell[BoxData[
 FormBox[GridBox[{
    {
     OverscriptBox[
      SubscriptBox["Y", "1"], "."], "=", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "1"], " ", 
       RowBox[{"(", 
        RowBox[{
         SubscriptBox["Y", "2"], "-", "1"}], " ", ")"}]}], " ", ","}], 
     OverscriptBox[
      SubscriptBox["Y", "2"], "."], "=", 
     RowBox[{
      SubscriptBox["Y", "2"], " ", 
      RowBox[{"(", 
       RowBox[{"2", " ", "-", " ", 
        SubscriptBox["Y", "1"]}], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"LotkaVolterraSystem",
 CellID->511886737],

Cell[TextData[{
 "where the dot denotes differentiation with respect to time ",
 StyleBox["T",
  FontSlant->"Italic"],
 "."
}], "Text",
 CellID->2070744103],

Cell[TextData[{
 "The Lotka-Volterra system (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSystem"],
 ") has an invariant ",
 Cell[BoxData[
  FormBox["H", TraditionalForm]]],
 ", which is constant for all ",
 StyleBox["T",
  FontSlant->"Italic"],
 ":"
}], "Text",
 CellID->1488710030],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"H", "(", 
    RowBox[{
     SubscriptBox["Y", "1"], ",", " ", 
     SubscriptBox["Y", "2"]}], ")"}], " ", "=", " ", 
   RowBox[{
    RowBox[{"2", " ", "ln", " ", 
     SubscriptBox["Y", "1"]}], " ", "-", " ", 
    SubscriptBox["Y", "1"], "+", " ", 
    RowBox[{"ln", " ", 
     SubscriptBox["Y", "2"]}], "-", " ", 
    SubscriptBox["Y", "2"]}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"LotkaVolterraInvariant",
 CellID->189413590],

Cell[TextData[{
 "The level curves of the invariant (",
 
 CounterBox["NumberedEquation", "LotkaVolterraInvariant"],
 ") are closed so that the solution is periodic."
}], "Text",
 CellID->1285984734],

Cell[TextData[{
 "It is desirable that the numerical solution of (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSystem"],
 ") is also periodic, but this is not always the case. "
}], "Text",
 CellID->785749523],

Cell[TextData[{
 "Note that (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSystem"],
 ") is a Poisson system:"
}], "Text",
 CellID->886862363],

Cell[BoxData[
 FormBox[GridBox[{
    {
     OverscriptBox["Y", "."], "=", 
     RowBox[{
      RowBox[{"B", "(", "Y", ")"}], " ", 
      RowBox[{"\[Del]", "H"}], 
      RowBox[{"(", "Y", ")"}]}], "=", 
     RowBox[{
      RowBox[{"(", GridBox[{
         {"0", 
          RowBox[{
           RowBox[{"-", 
            SubscriptBox["Y", "1"]}], 
           SubscriptBox["Y", "2"]}]},
         {
          RowBox[{
           SubscriptBox["Y", "1"], 
           SubscriptBox["Y", "2"]}], "0"}
        }], ")"}], 
      RowBox[{"(", GridBox[{
         {
          RowBox[{
           FractionBox["2", 
            SubscriptBox["Y", "1"]], "-", "1"}]},
         {
          RowBox[{
           FractionBox["1", 
            SubscriptBox["Y", "2"]], "-", "1"}]}
        }], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"PoissonSystemt",
 CellID->146782085],

Cell[TextData[{
 "where ",
 StyleBox["H",
  FontSlant->"Italic"],
 "(",
 StyleBox["Y",
  FontSlant->"Italic"],
 ") is defined in (",
 
 CounterBox["NumberedEquation", "LotkaVolterraInvariant"],
 ")."
}], "Text",
 CellID->281453157],

Cell[TextData[{
 "Poisson systems and Poisson integrators are discussed in Chapter VII.2 of \
[",
 ButtonBox["HLW02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2433"],
 "] and [",
 ButtonBox["MQ02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#31673"],
 "]."
}], "Text",
 CellID->1348394807],

Cell["\<\
Load a package with some predefined problems and select the Lotka-Volterra \
system.\
\>", "MathCaption",
 CellID->1940782007],

Cell[BoxData[{
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveProblems`\>\"", "]"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveUtilities`\>\"", "]"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{
   "Needs", "[", 
    "\"\<DifferentialEquations`InterpolatingFunctionAnatomy`\>\"", "]"}], 
   ";"}], "\n"}], "\n", 
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<LotkaVolterra\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"invts", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<Invariants\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"time", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<TimeData\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"step", " ", "=", " ", 
   RowBox[{"3", "/", "25"}]}], ";"}]}], "Input",
 InitializationCell->True,
 CellLabel->"In[10]:=",
 CellID->1949076620],

Cell["Define a utility function for visualizing solutions.", "MathCaption",
 CellID->940490592],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"LotkaVolterraPlot", "[", 
    RowBox[{"sol_", ",", "vars_", ",", "time_", ",", 
     RowBox[{"opts___", "?", "OptionQ"}]}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "data", ",", "data1", ",", "data2", ",", "ifuns", ",", "lplot", ",", 
       "pplot"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"ifuns", "=", 
       RowBox[{"First", "[", 
        RowBox[{"vars", "/.", "sol"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"data1", "=", 
       RowBox[{
        RowBox[{"Part", "[", 
         RowBox[{"ifuns", ",", "1", ",", "0"}], "]"}], "[", 
        "\"\<ValuesOnGrid\>\"", "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"data2", "=", 
       RowBox[{
        RowBox[{"Part", "[", 
         RowBox[{"ifuns", ",", "2", ",", "0"}], "]"}], "[", 
        "\"\<ValuesOnGrid\>\"", "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"data", "=", 
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"data1", ",", "data2"}], "}"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"commonopts", "=", 
       RowBox[{"Sequence", "[", 
        RowBox[{
         RowBox[{"Axes", "\[Rule]", "False"}], ",", 
         RowBox[{"Frame", "\[Rule]", "True"}], ",", 
         RowBox[{"FrameLabel", "\[Rule]", 
          RowBox[{"Join", "[", 
           RowBox[{
            RowBox[{"Map", "[", 
             RowBox[{"TraditionalForm", ",", "vars"}], "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"None", ",", "None"}], "}"}]}], "]"}]}], ",", 
         RowBox[{"RotateLabel", "\[Rule]", "False"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"lplot", "=", 
       RowBox[{"ListPlot", "[", 
        RowBox[{"data", ",", 
         RowBox[{"Evaluate", "[", 
          RowBox[{"FilterRules", "[", 
           RowBox[{
            RowBox[{"{", "opts", "}"}], ",", 
            RowBox[{"Options", "[", "ListPlot", "]"}]}], "]"}], "]"}], ",", 
         RowBox[{"PlotStyle", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"PointSize", "[", "0.02", "]"}], ",", 
            RowBox[{"RGBColor", "[", 
             RowBox[{"0", ",", "1", ",", "0"}], "]"}]}], "}"}]}], ",", 
         RowBox[{"Evaluate", "[", "commonopts", "]"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"pplot", "=", 
       RowBox[{"ParametricPlot", "[", 
        RowBox[{
         RowBox[{"Evaluate", "[", "ifuns", "]"}], ",", "time", ",", 
         RowBox[{"Evaluate", "[", 
          RowBox[{"FilterRules", "[", 
           RowBox[{
            RowBox[{"{", "opts", "}"}], ",", 
            RowBox[{"Options", "[", "ParametricPlot", "]"}]}], "]"}], "]"}], 
         ",", 
         RowBox[{"Evaluate", "[", "commonopts", "]"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"Show", "[", 
       RowBox[{"lplot", ",", "pplot"}], "]"}]}]}], "\[IndentingNewLine]", 
    "]"}]}], ";"}]], "Input",
 InitializationCell->True,
 CellLabel->"In[18]:=",
 CellID->307572160]
}, Open  ]],

Cell[CellGroupData[{

Cell["Explicit Euler", "Section",
 CellTags->"s:2",
 CellID->323310890],

Cell[TextData[{
 "Use the explicit or forward Euler method to solve the system ",
 "(",
 
 CounterBox["NumberedEquation", "LotkaVolterraSystem"],
 ")",
 "."
}], "MathCaption",
 CellID->1404604890],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"fesol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", "\"\<ExplicitEuler\>\""}], ",", 
      RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{"LotkaVolterraPlot", "[", 
  RowBox[{"fesol", ",", " ", "vars", ",", " ", "time"}], "]"}]}], "Input",
 CellLabel->"In[19]:=",
 CellID->1792836811],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->94028163]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Backward Euler", "Section",
 CellTags->"s:3",
 CellID->391572185],

Cell[TextData[{
 "Define the backward or implicit Euler method in terms of the RadauIIA \
implicit Runge-Kutta method and use it to solve (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSystem"],
 ") . ",
 "The resulting trajectory spirals from the initial conditions towards a \
fixed point at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"2", ",", "1"}], ")"}], TraditionalForm]]],
 " in a clockwise direction."
}], "MathCaption",
 CellID->2063362505],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"BackwardEuler", " ", "=", 
    RowBox[{"{", 
     RowBox[{"\"\<FixedStep\>\"", ",", 
      RowBox[{"Method", "\[Rule]", " ", 
       RowBox[{"{", 
        RowBox[{"\"\<ImplicitRungeKutta\>\"", ",", 
         RowBox[{
         "\"\<Coefficients\>\"", "\[Rule]", 
          "\"\<ImplicitRungeKuttaRadauIIACoefficients\>\""}], ",", 
         RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "1"}], ",", 
         RowBox[{"\"\<ImplicitSolver\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<FixedPoint\>\"", ",", 
            RowBox[{"AccuracyGoal", "\[Rule]", "MachinePrecision"}], ",", " ",
             
            RowBox[{"PrecisionGoal", "\[Rule]", "MachinePrecision"}], ",", 
            RowBox[{"\"\<IterationSafetyFactor\>\"", "\[Rule]", "1"}]}], 
           "}"}]}]}], "}"}]}]}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"besol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", "BackwardEuler"}], ",", 
      RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{"LotkaVolterraPlot", "[", 
  RowBox[{"besol", ",", " ", "vars", ",", " ", "time"}], "]"}]}], "Input",
 CellLabel->"In[21]:=",
 CellID->1972334912],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->591120652]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Projection", "Section",
 CellTags->"s:4",
 CellID->122851437],

Cell[TextData[{
 "Projection of the forward Euler method using the invariant (",
 
 CounterBox["NumberedEquation", "LotkaVolterraInvariant"],
 ") of the Lotka-Volterra equations gives a periodic solution."
}], "MathCaption",
 CellID->1973445976],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"pfesol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"Projection", ",", 
         RowBox[{"Method", "\[Rule]", "\"\<ExplicitEuler\>\""}], ",", " ", 
         RowBox[{"Invariants", "\[Rule]", "invts"}]}], "}"}]}], ",", 
      RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{"LotkaVolterraPlot", "[", 
  RowBox[{"pfesol", ",", " ", "vars", ",", " ", "time"}], "]"}]}], "Input",
 CellLabel->"In[24]:=",
 CellID->271854208],

Cell[GraphicsData["CompressedBitmap", "\<\
eJy9WXtsFMcdXm7v/ADzdowNhthACH4ABhzAFOpnAjbGxoAh4Bj8wHC8zgQu
PExA5hFHCkZpTFofansQYqgxtDgKPZr2SNpDaolUHVLlSJUbqXKl6PpHT5Vq
9Y+TKk2/mdm9W8/u+s6m9ATr3dmZ3/P7PWZ2c6PT3nK00XmwuTFjw/HGY/aD
zScy3mg9jiF5giRJD/H/ZKqEe4I79v8hIfirvR7GVcarN9jTEXqlj4T9JIvw
bBOe10RW2cKjbEySEoW5q9l4i8LParjGon2rW7tfGJUNachGs8ZJS5RDtAen
dYBerdq1FgM5xLXGtLRyUQnix08lIhFu4uifSYJ++qtUMMJLXAn+ixM4xeF2
LRHxNL5rtsLPYuiFRO1bre64nTjKuwTcWMfwjiksvtNJMxrVJDYnR3gjRk2C
8CzrvDUx6gwtjTBSR51hE54ThWfRv/FROUwy1FZW7CPG/3i0jE5DnCHa2ohC
bNKZa6e1SnQdxBnUjhaVvhjJo66MppuI5egcjHVUJTTSTTagG13n59dNHsVq
smnkmWki5vXoXozXjejjIZY5CboRmyKj8RutVNxb2cp8sTLEa8YtCsItSlzH
8jwxRrpcCn0OzyAmdQ+3H7AZL8dgHzpyGX8njGmFbMSXXsZc0cf1TLk8IA/m
kDmpJDXMJU472k26tePhQY1WL0bW/zlV1hfE7Bnq/RkCBRFhHOcw1WKymNJm
HBP/MviPZ/7vHnn+vMV9LsO9eZnjRKHjUqvj85rqT6X4wvXXGQX8pbc2fin9
Ed7zfxfavZLlhvuPku183+2F/rWZA/ncBdJMgfu0KPpSaSeMorNsEKnz6WUq
xC/ub8l01RQ4zu2svQMZ6+vuctnedz+scJ+q8Dfc8P8S04aG/qnknGfk2Xay
vYpUfUW+so0ceTT8WMZcKQ42gWb4t6njMsyy3H7aQvVPBOmW9pv57saTnm5M
vB/SYC93FC1E6UWEzKGXJA7l9ODCM17X1c4nzU33wRTXsg7qnnx/BRyWwQyq
1SEU+g9EqfU6Maewve2gvR+rVtd2LHE6XnFvPf+0Oxj8d3ZYNjWHqLlGlGQ2
cwgIVnZefK3uQkH1FdC81/cnsFb8Bt52Ygdj6VX6PD38jCfc1wztXewro7I4
LgFI8MrZti+AkAUDq6aRaYqtFoTlWabI8RKlNSkczBAbPtxUfgMW/0Hfoy2D
jVxZKZ9ZSsszEPjXY++3XNz1pR/ntB0s6t+HwUyFMueh1TGZXuLeChygwIGC
znb4e3g4VMiigtu2MLih3Gtv7eiFDsDUta7ffzPwd+bZCYY0WQTGwW4Qu6b2
ZkWPszKwEwJuoOMJQB9sCFeWV/84r3N/un8FN8RsU3pTGD1uD0TW7jYXVkPP
KlX29L7SlU3vws9F7qPQNk1n0akMaJiZ1b4PM4t8b21hsij6kcLKod1nXfeg
3ztOz9dP/5Ziaq/JLIXc7nlWW/dJpeeo4okdTBKIBMTBy82ey3EkbhfZNdNU
J9YD2KBT7tDarfbr111fA7tSHRsFvl5rPwZ1Znu+RxEyw5QK61usoJLhLYTL
iwa27WEUeklvSiDz1Y6GwrorT3x/labrLML4W7f2nUFopA/PhxJ7NYj+dMAj
WWAKSYa3FF5JpjbhvQSi9EznAw7FfUwKGJJCvuc2Mi3VI0kXeUbVW8wPNlVH
UGnouI5EZGfDAGlR50lkpEnBmZT6pFFyjnG10FeFElKCOFjpbpAOsRVw74yn
OdSy/h1jpx/J1eWeQ0BeHsnjdG0cd5uDtXX2W5/1f8MsaFbp1I7NMG9agZU9
9tsHQodg9aNsBnD0fsdvaTFKfC6LWFS7Z4Vyq5tcAMLbbARQPeXu4S6Ne34O
1ObIkNTmJ9jzmo635/VsTCEpUoS6ihe1exRtwSkhyyBo8kIrFUpzyVxkwHTf
Gg0lFf8ceee7PtvRd45GMFthRdzkhJZWNXXDrGo/oeA9PswNP5QRIJtG7ElG
B/kQkU+frSM4se6LL4Eh8137Drtdp9kKJFEkEXZrtgKGgTpZgXy+onf4F+tq
r3wY7BZWMBkp200De1CcATI239ZAGpAXF3btRB4iSmzpeKAz2Oq6CAOwNbT6
JwfTUQemh5ItI+WKdHYoQ7QxaONSkd41bSed3muS8Xyo/f3yLir1WcYB1QXo
5IQEHeSSYAVYLyKL+FwZqRxWFmRX6KKZ4cpK77K5KI9ICPQ2ghizkxqbhs4t
35eVbR9QBHD5TpPT+U3ntg3sD8uXKVDQ79a0SOf6MZmsaSQN6WmWf8kYafGz
sLCZFEujSoEafWUZi1yKnu957qzqaKV58qzq6XLHVfSDYdlojNDls8ZEt9jb
gMxGfcb9gMqMYIj4QTzXETN+LDtxc+5AAWwSQQEKOEIzwj1hJHIVGUQeav+n
zS4ckV8M/aGs9loYHTKIA2gRRHJh1J9NJ6vZCXPKSLnQpNLcwf2MiFnm24Qm
RxNRIor1uTXaObTAkbtJ4YhEs6Brx/K+N0WOhifmseR0gW8k3xzsuLWibxdg
eIbNQ31FilsRWK/hLHicjO98W+Bb7GlCNwSYcr73/L/baP+Q+tXYxiJyY+UY
zsVKKSJ5jJ8VjTDKfEQbI0vGemaeOlIzBPLrdR/9bPBXvEpEKlOkSnBgqj+i
oNfohD1tJG38UO3RSoRrFjp3MlrNwg+NAiQ4RSIZXF8XIx7daXcv9ZfRvPQO
e8YGCQ2rysO43+SWL/bVo7rTzMOrPY+bN53Xhb5BrAIMY6bIrQrRbRBFJKda
MrwR7d/iYJ6Gqph5rQJ1Y8pwFHqNU119CuXStgvJvjyau8wpi9lRPRPltoNd
YV26m+edGZyDhoZ34OE6oqMp+t0wt8ioY3AFvW1lz+jlUfepJxNMpY1GmWuz
vemnuYPrqcd5vwr5kY2AW02/GmNt09Eu9u9EV0AxcUTFH6TGle13zHvJSDwg
IyEXru5pZN8qJRlNBaSjO371xFKPx/DJCdrwbXU3oBDbodDeOTu4Ars7LI/s
fcTVPM8gGyv23a+ufSmQQfeqgzWTTdfyry7wO0KT3rawtbsHD2Nh5uAK4MB8
La+C6DJ4ZpKa2VoEH/YTzQEH1k4hYszyXSOk3Oz4KOXpMnSWjYwKUgTQQqWf
amolfuq9aDgXFTttMFvaq9oX+0y6WzHf5XKu6DSXBNbw3rxexWR93V1cJfN9
dpKKA8WLu1UNkNhgcMAuWaflFA12aKfLdve28PkDNkNo3dkpjUWfZXQ1U4xh
Rt3G9b7Yf0fZG29jctUHW1CmAOGioapZptFglBW0z1NVHYvaz77SWYdw2MqW
ASc1tTcr+48ppxbmHNTvVmKUsdNLuivGDm2d/RKsV82x7/8O9gWWMDJ7VKra
PGlT7Mao2vjJAzpS+BTZR6pklOEzIAubKwTIbGKW09S9mmgZdvpqha1BA+aG
dBs1VMEIcJ9nKq8ZVXY+Z3vs/ZafbmB/TD1YTEeVU6qXB/PQwOE1FMkgYuYR
OwvjXg1Sb+5qRxcxx7ea0i9iIu3wHUf80Lzg3Ttfh9xU1ZI5QwX8CJvidx0d
jYfygC1kOuDsrfa3KBgwPxlktOSwVyQLokGSeXGmr1apVuCHk4gnmmcXEbNI
TGdSQB5YHlJAlk8C9xX05zFaiGRACLRAEXSlLFNa85itwQ9cS6o/XuJ0zPWu
6xp2sS9y8beDP8/yVCxvP1JW7oLX+YEm+1o30lqZXCLyDH0DNjGF5dcwG0FC
K+pC+i4B6yAUmADdxeU/xKRUT0FRcOMSU6sxmpO51X4T+hLmutr5BECDxtCN
Zmr/3cg5OUPSNH62hWewg32Qj/ma10t/AoFgKPpxQnPKvdTULuwUOhF2gc6g
scFxhZ9/FzrbkWLpSZMFDPjuQfuxiNVACpWJsPspd092Vz2FT/k1iAC3gxTW
QopVOhuKX1umC88sJyrnrJVDu7t9n0MQWAJUEdQy/ZIQDyaQlG9u+VcXYEDi
77Cj459YtrjPIeks8Bc8CPxaF5EiV/ErS5JObsMe7f/3/U7PRen8AQFsThCf
eORfa+BNURrlm8MLlPmFUI144D12J034LwbWmwo=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->15635979]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Splitting", "Section",
 CellTags->"s:5",
 CellID->1463058597],

Cell[TextData[{
 "Another approach for obtaining the correct qualitative behavior is to \
additively split (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSystem"],
 ") into two systems"
}], "Text",
 CellID->1446422997],

Cell[BoxData[
 FormBox[GridBox[{
    {
     OverscriptBox[
      SubscriptBox["Y", "1"], "."], "=", 
     RowBox[{
      SubscriptBox["Y", "1"], " ", 
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["Y", "2"], " ", "-", " ", "1"}], " ", ")"}]}], 
     OverscriptBox[
      SubscriptBox["Y", "2"], "."], "=", "0"},
    {
     OverscriptBox[
      SubscriptBox["Y", "1"], "."], "=", "0", 
     OverscriptBox[
      SubscriptBox["Y", "2"], "."], "=", 
     RowBox[{
      SubscriptBox["Y", "2"], " ", 
      RowBox[{"(", 
       RowBox[{"2", " ", "-", " ", 
        SubscriptBox["Y", "1"]}], ")"}]}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"LotkaVolterraSplitting",
 CellID->263995033],

Cell[TextData[{
 "By appropriately solving (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSplitting"],
 ") it is possible to construct Poisson integrators."
}], "Text",
 CellID->1364149929],

Cell["\<\
Define the equations for splitting of the Lotka-Volterra equations.\
\>", "MathCaption",
 CellID->188986493],

Cell[BoxData[{
 RowBox[{
  RowBox[{"eqs", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<System\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"Y1", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y1", ",", " ", "2", ",", " ", "2"}], "]"}], " ", "=", " ", "0"}],
   ";"}], "\n", 
 RowBox[{
  RowBox[{"Y2", " ", "=", " ", "eqs"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Part", "[", 
    RowBox[{"Y2", ",", "1", ",", " ", "2"}], "]"}], " ", "=", " ", "0"}], 
  ";"}]}], "Input",
 CellLabel->"In[26]:=",
 CellID->775881770],

Cell[CellGroupData[{

Cell["Symplectic Euler", "Subsection",
 CellID->1287410384],

Cell[TextData[{
 "Define the symplectic Euler method in terms of a splitting method using the \
backward and forward Euler methods for each system in (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSplitting"],
 ")."
}], "MathCaption",
 CellID->1677696523],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"SymplecticEuler", "=", " ", 
    RowBox[{"{", 
     RowBox[{"\"\<Splitting\>\"", ",", "\[IndentingNewLine]", " ", 
      RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "1"}], ",", 
      RowBox[{"\"\<Equations\>\"", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"Y1", ",", "Y2"}], "}"}]}], ",", " ", 
      RowBox[{"\"\<Method\>\"", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"BackwardEuler", ",", "\"\<ExplicitEuler\>\""}], "}"}]}]}], 
     "}"}]}], ";"}], "\n"}], "\n", 
 RowBox[{
  RowBox[{"sesol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", 
     RowBox[{"Method", "\[Rule]", "SymplecticEuler"}], ",", 
     RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[31]:=",
 CellID->965295360],

Cell["\<\
The numerical solution using the symplectic Euler method is periodic.\
\>", "MathCaption",
 CellID->425911471],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LotkaVolterraPlot", "[", 
  RowBox[{"sesol", ",", " ", "vars", ",", " ", "time"}], "]"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->315962398],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]=",
 CellID->630697267]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Flows", "Subsection",
 CellID->572693556],

Cell[TextData[{
 "Consider splitting the Lotka-Volterra equations and computing the flow (or \
exact solution) of each system in (",
 
 CounterBox["NumberedEquation", "LotkaVolterraSplitting"],
 ")."
}], "Text",
 CellID->597415555],

Cell["\<\
The solutions can be found as follows, where the constants should be related \
to the initial conditions at each step.\
\>", "Text",
 CellID->82414739],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{"Y1", ",", "vars", ",", "T"}], "]"}]], "Input",
 CellLabel->"In[34]:=",
 CellID->35541743],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{"C", "[", "1", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"T", " ", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"-", "1"}], "+", 
          RowBox[{"C", "[", "1", "]"}]}], ")"}]}]], " ", 
      RowBox[{"C", "[", "2", "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{261, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]=",
 CellID->2106953525]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DSolve", "[", 
  RowBox[{"Y2", ",", "vars", ",", "T"}], "]"}]], "Input",
 CellLabel->"In[35]:=",
 CellID->105303117],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{"C", "[", "1", "]"}]}], ",", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "2"], "[", "T", "]"}], "\[Rule]", 
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"T", " ", 
        RowBox[{"(", 
         RowBox[{"2", "-", 
          RowBox[{"C", "[", "1", "]"}]}], ")"}]}]], " ", 
      RowBox[{"C", "[", "2", "]"}]}]}]}], "}"}], "}"}]], "Output",
 ImageSize->{256, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]=",
 CellID->1941389516]
}, Open  ]],

Cell[TextData[{
 "An advantage of locally computing the flow is that it yields an ",
 StyleBox["explicit,",
  FontSlant->"Italic"],
 " and hence very efficient, integration procedure."
}], "Text",
 CellID->18342651],

Cell[TextData[{
 "The ",
 StyleBox[ButtonBox["LocallyExact",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveLocallyExact"], "MR"],
 " method provides a general way of computing the flow of each splitting \
using ",
 Cell[BoxData[
  ButtonBox["DSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/DSolve"]], "InlineFormula"],
 " ",
 "only during the initialization phase."
}], "Text",
 CellID->365991916],

Cell["\<\
Set up a hybrid symbolic-numeric splitting method and use it to solve the \
Lotka-Volterra system.\
\>", "MathCaption",
 CellID->13764180],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"SplittingLotkaVolterra", "=", " ", 
    RowBox[{"{", 
     RowBox[{"\"\<Splitting\>\"", ",", "\[IndentingNewLine]", " ", 
      RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "1"}], ",", 
      RowBox[{"\"\<Equations\>\"", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"Y1", ",", "Y2"}], "}"}]}], ",", " ", 
      RowBox[{"\"\<Method\>\"", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<LocallyExact\>\"", ",", "\"\<LocallyExact\>\""}], 
        "}"}]}]}], "}"}]}], ";"}], "\n"}], "\n", 
 RowBox[{
  RowBox[{"spsol", " ", "=", " ", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"system", ",", 
     RowBox[{"Method", "\[Rule]", "SplittingLotkaVolterra"}], ",", 
     RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[36]:=",
 CellID->399008527],

Cell["\<\
The numerical solution using the splitting method is periodic.\
\>", "MathCaption",
 CellID->1993559475],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"LotkaVolterraPlot", "[", 
  RowBox[{"spsol", ",", " ", "vars", ",", " ", "time"}], "]"}]], "Input",
 CellLabel->"In[38]:=",
 CellID->902194100],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->67797398]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Numerical methods for solving the Lotka-Volterra equations - \
Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 55.8683872}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "The Lotka-Volterra system arises in mathematical biology and models the \
growth of animal species. Consider two species where Y_1(T)denotes the number \
of predators and Y_2(T) denotes the number of prey. A particular case of the \
Lotka-Volterra differential system is:", "synonyms" -> {}, "title" -> 
    "Numerical methods for solving the Lotka-Volterra equations", "type" -> 
    "Tutorial", "uri" -> "tutorial/NDSolveLotkaVolterra"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "s:1"->{
  Cell[849, 34, 70, 2, 70, "Section",
   CellTags->"s:1",
   CellID->1174568789]},
 "LotkaVolterraSystem"->{
  Cell[1562, 65, 587, 20, 70, "NumberedEquation",
   CellTags->"LotkaVolterraSystem",
   CellID->511886737]},
 "LotkaVolterraInvariant"->{
  Cell[2603, 109, 489, 15, 70, "NumberedEquation",
   CellTags->"LotkaVolterraInvariant",
   CellID->189413590]},
 "PoissonSystemt"->{
  Cell[3665, 150, 871, 32, 70, "NumberedEquation",
   CellTags->"PoissonSystemt",
   CellID->146782085]},
 "s:2"->{
  Cell[9534, 341, 71, 2, 70, "Section",
   CellTags->"s:2",
   CellID->323310890]},
 "s:3"->{
  Cell[15788, 470, 71, 2, 70, "Section",
   CellTags->"s:3",
   CellID->391572185]},
 "s:4"->{
  Cell[23319, 628, 67, 2, 70, "Section",
   CellTags->"s:4",
   CellID->122851437]},
 "s:5"->{
  Cell[28684, 741, 67, 2, 70, "Section",
   CellTags->"s:5",
   CellID->1463058597]},
 "LotkaVolterraSplitting"->{
  Cell[28980, 754, 709, 24, 70, "NumberedEquation",
   CellTags->"LotkaVolterraSplitting",
   CellID->263995033]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"s:1", 46812, 1222},
 {"LotkaVolterraSystem", 46921, 1226},
 {"LotkaVolterraInvariant", 47060, 1230},
 {"PoissonSystemt", 47195, 1234},
 {"s:2", 47311, 1238},
 {"s:3", 47405, 1242},
 {"s:4", 47500, 1246},
 {"s:5", 47595, 1250},
 {"LotkaVolterraSplitting", 47710, 1254}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 103, 2, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[728, 29, 96, 1, 70, "Title",
 CellID->1637096269],
Cell[CellGroupData[{
Cell[849, 34, 70, 2, 70, "Section",
 CellTags->"s:1",
 CellID->1174568789],
Cell[922, 38, 144, 4, 70, "Text",
 CellID->1832907302],
Cell[1069, 44, 380, 14, 70, "Text",
 CellID->2048574980],
Cell[1452, 60, 107, 3, 70, "Text",
 CellID->890047143],
Cell[1562, 65, 587, 20, 70, "NumberedEquation",
 CellTags->"LotkaVolterraSystem",
 CellID->511886737],
Cell[2152, 87, 156, 6, 70, "Text",
 CellID->2070744103],
Cell[2311, 95, 289, 12, 70, "Text",
 CellID->1488710030],
Cell[2603, 109, 489, 15, 70, "NumberedEquation",
 CellTags->"LotkaVolterraInvariant",
 CellID->189413590],
Cell[3095, 126, 199, 6, 70, "Text",
 CellID->1285984734],
Cell[3297, 134, 215, 6, 70, "Text",
 CellID->785749523],
Cell[3515, 142, 147, 6, 70, "Text",
 CellID->886862363],
Cell[3665, 150, 871, 32, 70, "NumberedEquation",
 CellTags->"PoissonSystemt",
 CellID->146782085],
Cell[4539, 184, 231, 12, 70, "Text",
 CellID->281453157],
Cell[4773, 198, 344, 12, 70, "Text",
 CellID->1348394807],
Cell[5120, 212, 136, 4, 70, "MathCaption",
 CellID->1940782007],
Cell[5259, 218, 1042, 34, 70, "Input",
 InitializationCell->True,
 CellID->1949076620],
Cell[6304, 254, 95, 1, 70, "MathCaption",
 CellID->940490592],
Cell[6402, 257, 3095, 79, 70, "Input",
 InitializationCell->True,
 CellID->307572160]
}, Open  ]],
Cell[CellGroupData[{
Cell[9534, 341, 71, 2, 70, "Section",
 CellTags->"s:2",
 CellID->323310890],
Cell[9608, 345, 196, 8, 70, "MathCaption",
 CellID->1404604890],
Cell[CellGroupData[{
Cell[9829, 357, 461, 12, 65, "Input",
 CellID->1792836811],
Cell[10293, 371, 5446, 93, 139, "Output",
 Evaluatable->False,
 CellID->94028163]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[15788, 470, 71, 2, 70, "Section",
 CellTags->"s:3",
 CellID->391572185],
Cell[15862, 474, 470, 14, 70, "MathCaption",
 CellID->2063362505],
Cell[CellGroupData[{
Cell[16357, 492, 1337, 33, 155, "Input",
 CellID->1972334912],
Cell[17697, 527, 5573, 95, 139, "Output",
 Evaluatable->False,
 CellID->591120652]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[23319, 628, 67, 2, 70, "Section",
 CellTags->"s:4",
 CellID->122851437],
Cell[23389, 632, 245, 6, 70, "MathCaption",
 CellID->1973445976],
Cell[CellGroupData[{
Cell[23659, 642, 633, 16, 101, "Input",
 CellID->271854208],
Cell[24295, 660, 4340, 75, 139, "Output",
 Evaluatable->False,
 CellID->15635979]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[28684, 741, 67, 2, 70, "Section",
 CellTags->"s:5",
 CellID->1463058597],
Cell[28754, 745, 223, 7, 70, "Text",
 CellID->1446422997],
Cell[28980, 754, 709, 24, 70, "NumberedEquation",
 CellTags->"LotkaVolterraSplitting",
 CellID->263995033],
Cell[29692, 780, 194, 6, 70, "Text",
 CellID->1364149929],
Cell[29889, 788, 118, 3, 70, "MathCaption",
 CellID->188986493],
Cell[30010, 793, 579, 19, 70, "Input",
 CellID->775881770],
Cell[CellGroupData[{
Cell[30614, 816, 59, 1, 70, "Subsection",
 CellID->1287410384],
Cell[30676, 819, 260, 7, 70, "MathCaption",
 CellID->1677696523],
Cell[30939, 828, 820, 22, 70, "Input",
 CellID->965295360],
Cell[31762, 852, 120, 3, 70, "MathCaption",
 CellID->425911471],
Cell[CellGroupData[{
Cell[31907, 859, 167, 4, 28, "Input",
 CellID->315962398],
Cell[32077, 865, 4504, 78, 139, "Output",
 Evaluatable->False,
 CellID->630697267]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[36630, 949, 47, 1, 70, "Subsection",
 CellID->572693556],
Cell[36680, 952, 231, 7, 70, "Text",
 CellID->597415555],
Cell[36914, 961, 161, 4, 70, "Text",
 CellID->82414739],
Cell[CellGroupData[{
Cell[37100, 969, 139, 4, 70, "Input",
 CellID->35541743],
Cell[37242, 975, 686, 23, 39, "Output",
 CellID->2106953525]
}, Open  ]],
Cell[CellGroupData[{
Cell[37965, 1003, 140, 4, 70, "Input",
 CellID->105303117],
Cell[38108, 1009, 660, 22, 39, "Output",
 CellID->1941389516]
}, Open  ]],
Cell[38783, 1034, 215, 6, 70, "Text",
 CellID->18342651],
Cell[39001, 1042, 420, 14, 70, "Text",
 CellID->365991916],
Cell[39424, 1058, 148, 4, 70, "MathCaption",
 CellID->13764180],
Cell[39575, 1064, 843, 22, 70, "Input",
 CellID->399008527],
Cell[40421, 1088, 114, 3, 70, "MathCaption",
 CellID->1993559475],
Cell[CellGroupData[{
Cell[40560, 1095, 167, 4, 28, "Input",
 CellID->902194100],
Cell[40730, 1101, 4466, 77, 139, "Output",
 Evaluatable->False,
 CellID->67797398]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[45247, 1184, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

