(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     75494,       2772]
NotebookOptionsPosition[     64141,       2402]
NotebookOutlinePosition[     67682,       2497]
CellTagsIndexPosition[     67488,       2487]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], " "}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["OrthogonalProjection Method for NDSolve", "Title",
 CellTags->"c:1",
 CellID->569226817],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->293332788],

Cell["Consider the matrix differential equation:", "Text",
 CellID->2028167128],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox["y", "\[Prime]"], "(", "t", ")"}], "=", 
    RowBox[{"f", "(", 
     RowBox[{"t", ",", 
      RowBox[{"y", "(", "t", ")"}]}], ")"}]}], ",", 
   RowBox[{"t", ">", "0"}], ","}], TraditionalForm]], "DisplayMath",
 CellID->1060194370],

Cell[TextData[{
 "where the initial value ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["y", "0"], "=", " ", 
    RowBox[{
     RowBox[{"y", "(", "0", ")"}], Cell[""]}]}], TraditionalForm]]],
 "\[Element] ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", 
    RowBox[{"m", "\[Times]", "p"}]], TraditionalForm]]],
 " is given. Assume that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      SubscriptBox["y", "0"], "T"], 
     SubscriptBox["y", "0"]}], " ", "=", " ", "I"}], TraditionalForm]]],
 ", that the solution has the property of preserving orthonormality, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"y", "(", "t", ")"}], "T"], " ", 
     RowBox[{"y", "(", "t", ")"}]}], " ", "=", " ", "I"}], 
   TraditionalForm]]],
 ", and that it has full rank for all t \[GreaterEqual] 0."
}], "Text",
 CellID->22383832],

Cell[TextData[{
 "From a numerical perspective, a key issue is how to numerically integrate \
an orthogonal matrix differential system in such a way that the numerical \
solution remains orthogonal. There are several strategies that are possible. \
One approach, suggested in [",
 ButtonBox["DRV94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#3420"],
 "], is to use an implicit Runge-Kutta method (such as the Gauss scheme). \
Some alternative strategies are described in [",
 ButtonBox["DV99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2646"],
 "] and [",
 ButtonBox["DL01",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2703"],
 "]."
}], "Text",
 CellID->1321681495],

Cell["\<\
The approach which we will take up here is to use any reasonable numerical \
integration method and then post-process using a projective procedure at the \
end of each integration step.\
\>", "Text",
 CellID->2115220046],

Cell[TextData[{
 "An important feature of our implementation is that the basic integration \
method can be any built-in numerical method, or even a user-defined \
procedure. In the following examples an explicit Runge-Kutta method is used \
for the basic time stepping. However, if we require greater accuracy we could \
easily use an extrapolation method, for example, by simply setting the \
appropriate ",
 StyleBox["Method", "MR"],
 " option."
}], "Text",
 CellID->1504773441],

Cell[CellGroupData[{

Cell["Projection step", "Subsection",
 CellID->1383769894],

Cell[TextData[{
 "At the end of each numerical integration step we need to transform the \
approximate solution matrix of the differential system to obtain an \
orthogonal matrix. This can be carried out in several ways (see for example \
[",
 ButtonBox["DRV94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#3420"],
 "] and [",
 ButtonBox["H97",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#20646"],
 "]):"
}], "Text",
 CellID->2138858796],

Cell["Newton or Schulz iteration", "BulletedText",
 CellID->1478293461],

Cell["QR decomposition", "BulletedText",
 CellID->1826935855],

Cell[CellGroupData[{

Cell["Singular value decomposition", "BulletedText",
 CellID->938809808],

Cell["\<\
The Newton and Schulz methods are quadratically convergent, and the number of \
iterations may vary depending on the error tolerances used in the numerical \
integration. One or two iterations are usually sufficient for convergence to \
the orthonormal polar factor (see below) in IEEE double-precision arithmetic.\
\
\>", "Text",
 CellID->353456932],

Cell["\<\
QR decomposition is cheaper than singular value decomposition (roughly by a \
factor of two), but it does not give the closest possible projection.\
\>", "Text",
 CellID->1993458239],

Cell[TextData[{
 StyleBox["Definition",
  FontWeight->"Bold"],
 StyleBox[" (Thin singular value decomposition ",
  FontVariations->{"CompatibilityType"->0}],
 "[",
 ButtonBox["GVL96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#18126"],
 "]",
 StyleBox["):",
  FontVariations->{"CompatibilityType"->0}],
 " Given a matrix ",
 Cell[BoxData[
  FormBox[
   RowBox[{"A", "\[Element]", 
    SuperscriptBox["\[DoubleStruckCapitalR]", 
     RowBox[{"m", "\[Times]", "p"}]]}], TraditionalForm]]],
 " with ",
 StyleBox["m",
  FontSlant->"Italic"],
 " \[GreaterEqual] ",
 StyleBox["p",
  FontSlant->"Italic"],
 " there exist two matrices U \[Element] ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", 
    RowBox[{"m", "\[Times]", "p"}]], TraditionalForm]]],
 " and ",
 StyleBox["V",
  FontSlant->"Italic"],
 " \[Element] ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", 
    RowBox[{"p", "\[Times]", "p"}]], TraditionalForm]]],
 " such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["U", "T"], " ", "A", " ", "V", " "}], TraditionalForm]]],
 " is the diagonal matrix of singular values of ",
 StyleBox["A",
  FontSlant->"Italic"],
 ", \[CapitalSigma] = ",
 Cell[BoxData[
  FormBox[
   RowBox[{"diag", "(", 
    RowBox[{
     SubscriptBox["\[Sigma]", "1"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["\[Sigma]", "p"]}], ")"}], TraditionalForm]]],
 "\[Element] ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", 
    RowBox[{"p", "\[Times]", "p"}]], TraditionalForm]]],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Sigma]", "1"], "\[GreaterEqual]", "\[CenterEllipsis]", 
    "\[GreaterEqual]", 
    SubscriptBox["\[Sigma]", "p"], "\[GreaterEqual]", "0"}], 
   TraditionalForm]]],
 ". ",
 StyleBox["U",
  FontSlant->"Italic"],
 " has orthonormal columns and ",
 StyleBox["V",
  FontSlant->"Italic"],
 " is orthogonal."
}], "Text",
 CellID->1838466452],

Cell[TextData[{
 StyleBox["Definition",
  FontWeight->"Bold"],
 StyleBox[" (Polar decomposition):",
  FontVariations->{"CompatibilityType"->0}],
 " Given a matrix ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]]],
 " and its singular value decomposition ",
 Cell[BoxData[
  FormBox[
   RowBox[{"U", " ", "\[CapitalSigma]", " ", 
    FormBox[
     SuperscriptBox["V", "T"],
     TraditionalForm]}], TraditionalForm]]],
 ", the polar decomposition of A is given by the product of two matrices ",
 StyleBox["Z",
  FontSlant->"Italic"],
 " and ",
 StyleBox["P",
  FontSlant->"Italic"],
 " where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"Z", "=", 
    RowBox[{"U", " ", 
     SuperscriptBox["V", "T"]}]}], TraditionalForm]]],
 " and ",
 StyleBox["P",
  FontSlant->"Italic"],
 " = ",
 Cell[BoxData[
  FormBox[
   RowBox[{"V", " ", "\[CapitalSigma]", " ", 
    SuperscriptBox["V", "T"]}], TraditionalForm]]],
 ". ",
 StyleBox["Z",
  FontSlant->"Italic"],
 " has orthonormal columns and ",
 StyleBox["P",
  FontSlant->"Italic"],
 " is symmetric positive semidefinite."
}], "Text",
 CellID->843430310],

Cell["\<\
The orthonormal polar factor Z of A is the matrix that solves:\
\>", "Text",
 CellID->760825666],

Cell[BoxData[
 FormBox[
  RowBox[{
   UnderscriptBox["min", 
    RowBox[{"Z", " ", "\[Element]", " ", 
     FormBox[
      SuperscriptBox["\[DoubleStruckCapitalR]", 
       RowBox[{"m", "\[Times]", "p"}]],
      TraditionalForm]}]], 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"||", 
       RowBox[{"A", "-", "Z"}], "||"}], " ", ":", " ", 
      RowBox[{
       FormBox[
        SuperscriptBox["Z", "T"],
        TraditionalForm], "Z"}]}], "=", "I"}], "}"}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->1907055565],

Cell[TextData[{
 "for the 2 and Frobenius norms [",
 ButtonBox["H96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#12862"],
 "]."
}], "Text",
 CellID->1669934484]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Schulz iteration", "Subsection",
 CellID->111723291],

Cell[TextData[{
 "We have chosen an approach based on the Schulz iteration, which works \
directly for ",
 StyleBox["m",
  FontSlant->"Italic"],
 " >= ",
 StyleBox["p",
  FontSlant->"Italic"],
 ". In contrast Newton iteration for ",
 StyleBox["m",
  FontSlant->"Italic"],
 " > ",
 StyleBox["p",
  FontSlant->"Italic"],
 " needs to be preceded by QR decomposition."
}], "Text",
 CellID->233496521],

Cell["\<\
Comparison with direct computation based on the singular value decomposition \
is also given.\
\>", "Text",
 CellID->1793217278],

Cell["The Schulz iteration is given by:", "Text",
 CellID->252956962],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["Y", 
     RowBox[{"i", "+", "1"}]], "=", 
    RowBox[{
     SubscriptBox["Y", "i"], "+", 
     RowBox[{
      RowBox[{
       SubscriptBox["Y", "i"], "(", 
       RowBox[{"I", "-", 
        RowBox[{
         SubsuperscriptBox["Y", "i", "T"], " ", 
         SubscriptBox["Y", "i"]}]}], ")"}], "/", "2"}]}]}], ",", " ", 
   RowBox[{
    SubscriptBox["Y", "0"], "=", 
    RowBox[{"A", "."}]}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"SchulzIteration",
 CellID->1828686581],

Cell[TextData[{
 "The Schulz iteration has an arithmetic operation count per iteration of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", 
     SuperscriptBox["m", "2"], "p"}], "+", 
    RowBox[{"2", "m", " ", 
     SuperscriptBox["p", "2"]}]}], TraditionalForm]]],
 " floating-point operations, but is rich in matrix multiplication [",
 ButtonBox["H97",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#20646"],
 "]."
}], "Text",
 CellID->182586918],

Cell[TextData[{
 "In a practical implementation, gemm level 3 BLAS of LAPACK [",
 ButtonBox["LAPACK99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#18138"],
 "] can be used in conjunction with architecture-specific optimizations via \
the Automatically Tuned Linear Algebra Software [",
 ButtonBox["ATLAS00",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#21834"],
 "]. Such considerations mean that the arithmetic operation count of the \
Schulz iteration is not necessarily an accurate reflection of the observed \
computational cost. A useful bound on the departure from orthonormality of A \
is in [",
 ButtonBox["H89",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#7204"],
 "]: ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     RowBox[{
      SuperscriptBox["A", "T"], " ", "A"}], " ", "-", " ", "I"}], 
    SubscriptBox["||", "F"]}], TraditionalForm]]],
 ". Comparison with the Schulz iteration gives the stopping criterion ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", 
    RowBox[{
     RowBox[{
      SuperscriptBox["A", "T"], " ", "A"}], " ", "-", " ", "I"}], 
    SubscriptBox["||", "F"], 
    RowBox[{"<", "\[Tau]"}]}], TraditionalForm]]],
 " for some tolerance \[Tau]."
}], "Text",
 CellID->86620304]
}, Open  ]],

Cell[CellGroupData[{

Cell["Standard formulation", "Subsection",
 CellID->378913166],

Cell[TextData[{
 "Assume that an initial value ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "n"], TraditionalForm]]],
 " for the current solution of the ODE is given, together with a solution ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["y", 
     RowBox[{"n", "+", "1"}]], "=", 
    RowBox[{
     SubscriptBox["y", "n"], "+", 
     SubscriptBox["\[CapitalDelta]y", "n"]}]}], TraditionalForm]]],
 " from a one-step numerical integration method. Assume that an absolute \
tolerance \[Tau] for controlling the Schulz iteration is also prescribed."
}], "Text",
 CellID->875969885],

Cell["The following algorithm can be used for implementation.", "Text",
 CellID->1270581329],

Cell[TextData[{
 "Step 1.",
 " ",
 "Set ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", "0"], "=", 
    SubscriptBox["y", 
     RowBox[{"n", "+", "1"}]]}], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"i", "=", "0"}], TraditionalForm]]],
 ".\nStep 2. Compute ",
 Cell[BoxData[
  FormBox[
   RowBox[{"E", " ", "=", " ", 
    RowBox[{"I", " ", "-", " ", 
     RowBox[{
      SubsuperscriptBox["Y", "i", "T"], " ", 
      SubscriptBox["Y", "i"]}]}]}], TraditionalForm]]],
 ".\nStep 3. Compute ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", 
     RowBox[{"i", "+", "1"}]], "=", 
    RowBox[{
     SubscriptBox["Y", "i"], "+", 
     RowBox[{
      SubscriptBox["Y", "i"], 
      RowBox[{"E", "/", "2"}]}]}]}], TraditionalForm]]],
 ".\nStep 4. If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", "E", 
    SubscriptBox["||", "F"], 
    RowBox[{"\[LessEqual]", "\[Tau]"}]}], TraditionalForm]]],
 " or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"i", "=", "imax"}], ","}], TraditionalForm]]],
 " then return ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["Y", 
    RowBox[{"i", "+", "1"}]], TraditionalForm]]],
 ".\nStep 5. Set ",
 Cell[BoxData[
  FormBox[
   RowBox[{"i", "=", " ", 
    RowBox[{"i", " ", "+", " ", "1"}]}], TraditionalForm]]],
 " and go to Step 2."
}], "Text",
 CellID->1272425816]
}, Open  ]],

Cell[CellGroupData[{

Cell["Increment formulation", "Subsection",
 CellID->1445196703],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " uses compensated summation to reduce the effect of rounding errors made by \
repeatedly adding the contribution of small quantities ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalDelta]y", "n"], TraditionalForm]]],
 "to ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["y", "n"], TraditionalForm]]],
 " at each integration step [",
 ButtonBox["H96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#12862"],
 "]. Therefore, the increment ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalDelta]y", "n"], TraditionalForm]]],
 " is returned by the base integrator."
}], "Text",
 CellID->1563914118],

Cell[TextData[{
 "An appropriate orthogonal correction ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalDelta]Y", "i"], TraditionalForm]]],
 " for the projective iteration can be determined using the following \
algorithm."
}], "Text",
 CellID->1153109297],

Cell[TextData[{
 "Step 1. Set ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalDelta]Y", "0"], "=", "0"}], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"i", "=", "0"}], TraditionalForm]]],
 ".\nStep 2. Set ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", "i"], "=", 
    RowBox[{
     SubscriptBox["\[CapitalDelta]Y", "i"], "+", 
     SubscriptBox["y", 
      RowBox[{"n", "+", "1"}]]}]}], TraditionalForm]]],
 ".\nStep 3. Compute ",
 Cell[BoxData[
  FormBox[
   RowBox[{"E", " ", "=", " ", 
    RowBox[{"I", " ", "-", " ", 
     RowBox[{
      SubsuperscriptBox["Y", "i", "T"], " ", 
      SubscriptBox["Y", "i"]}]}]}], TraditionalForm]]],
 ".\nStep 4. Compute ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalDelta]Y", 
     RowBox[{"i", "+", "1"}]], "=", 
    RowBox[{
     SubscriptBox["\[CapitalDelta]Y", "i"], "+", 
     RowBox[{
      SubscriptBox["Y", "i"], 
      RowBox[{"E", "/", "2"}]}]}]}], TraditionalForm]]],
 ".\nStep 5. If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", "E", 
    SubscriptBox["||", "F"], 
    RowBox[{"\[LessEqual]", "\[Tau]"}]}], TraditionalForm]]],
 " or ",
 Cell[BoxData[
  FormBox[
   RowBox[{"i", "=", "imax"}], TraditionalForm]]],
 ", then return ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CapitalDelta]Y", 
     RowBox[{"i", "+", "1"}]], "+", 
    SubscriptBox["\[CapitalDelta]y", "n"]}], TraditionalForm]]],
 ".\nStep 6. Set ",
 Cell[BoxData[
  FormBox[
   RowBox[{"i", "=", " ", 
    RowBox[{"i", " ", "+", " ", "1"}]}], TraditionalForm]]],
 " and go to Step 2."
}], "Text",
 CellID->357907425],

Cell[TextData[{
 "This modified algorithm is used in ",
 StyleBox["OrthogonalProjection ", "MR"],
 "and shows an advantage of using an iterative process over a direct process, \
since it is not obvious how an orthogonal correction can be derived for \
direct methods."
}], "Text",
 CellID->1586297950]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
 CellTags->"s:2",
 CellID->327307144],

Cell[CellGroupData[{

Cell["Orthogonal error measurement", "Subsection",
 CellID->325643823],

Cell[TextData[{
 "A function to compute the Frobenius norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"||", "A", 
    SubscriptBox["||", "F"]}], TraditionalForm]]],
 " of a matrix ",
 StyleBox["A",
  FontSlant->"Italic"],
 " can be defined in terms of the ",
 StyleBox["Norm", "MR"],
 " function as follows."
}], "Text",
 CellID->943587744],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"FrobeniusNorm", "[", 
    RowBox[{"a_", "?", "MatrixQ"}], "]"}], ":=", " ", 
   RowBox[{"Norm", "[", 
    RowBox[{"a", ",", " ", "Frobenius"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1711077038],

Cell[TextData[{
 "An upper bound on the departure from orthonormality of ",
 StyleBox["A",
  FontSlant->"Italic"],
 " can then be measured using this function [",
 ButtonBox["H97",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#20646"],
 "]."
}], "Text",
 CellID->317018971],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"OrthogonalError", "[", 
    RowBox[{"a_", "?", "MatrixQ"}], "]"}], ":=", " ", 
   RowBox[{"FrobeniusNorm", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Transpose", "[", "a", "]"}], ".", "a"}], " ", "-", " ", 
     RowBox[{"IdentityMatrix", "[", 
      RowBox[{"Last", "[", 
       RowBox[{"Dimensions", "[", "a", "]"}], "]"}], "]"}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->274397557],

Cell["\<\
This defines the utility function for visualizing the orthogonal error during \
a numerical integration.\
\>", "Text",
 CellID->1390529246],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{
   "Utility", " ", "function", " ", "for", " ", "extracting", " ", "a", " ", 
    "list", " ", "of", " ", "values", " ", "of", " ", "the", " ", 
    "independent", " ", "variable", " ", "at", " ", "which", " ", "the", " ", 
    "integration", " ", "method", " ", "has", " ", "sampled"}], " ", "*)"}], 
  "\[IndentingNewLine]", "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"TimeData", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"v_", "?", "VectorQ"}], ",", 
        RowBox[{"___", "?", "VectorQ"}]}], "}"}], "]"}], ":=", 
     RowBox[{"TimeData", "[", "v", "]"}]}], ";"}], "\[IndentingNewLine]", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     RowBox[{"TimeData", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"if", ":", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"InterpolatingFunction", "[", "__", "]"}], ")"}], "[", "_",
           "]"}]}], ",", "___"}], "}"}], "]"}], ":=", "\[IndentingNewLine]", 
     RowBox[{"Part", "[", 
      RowBox[{"if", ",", "0", ",", "3", ",", "1"}], "]"}]}], 
    ";"}]}]}]], "Input",
 CellLabel->"In[4]:=",
 CellID->1255828779],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{
   "Utility", " ", "function", " ", "for", " ", "plotting", " ", "the", " ", 
    "orthogonal", " ", "error", " ", "in", " ", "a", " ", "numerical", " ", 
    "integration"}], " ", "*)"}], "\[IndentingNewLine]", 
  "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"OrthogonalErrorPlot", "[", "sol_", "]"}], ":=", 
    "\[IndentingNewLine]", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"errdata", ",", " ", "samples", ",", " ", "soldata"}], "}"}], 
      ",", "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{
       "Form", " ", "a", " ", "list", " ", "of", " ", "times", " ", "at", " ",
         "which", " ", "the", " ", "method", " ", "is", " ", "invoked"}], " ",
        "*)"}], "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"samples", " ", "=", " ", 
        RowBox[{"TimeData", "[", "sol", "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
        "Form", " ", "a", " ", "list", " ", "of", " ", "solutions", " ", "at",
          " ", "the", " ", "integration", " ", "times"}], " ", "*)"}], 
       "\[IndentingNewLine]", 
       RowBox[{"soldata", "=", 
        RowBox[{"Map", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{"sol", "/.", 
             RowBox[{"t", "\[Rule]", "#"}]}], ")"}], "&"}], ",", "samples"}], 
         "]"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"(*", " ", 
        RowBox[{
        "Form", " ", "a", " ", "list", " ", "of", " ", "the", " ", 
         "orthogonal", " ", "errors"}], " ", "*)"}], "\[IndentingNewLine]", 
       RowBox[{"errdata", " ", "=", " ", 
        RowBox[{"Map", "[", 
         RowBox[{"OrthogonalError", ",", " ", "soldata"}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"ListLinePlot", "[", 
        RowBox[{
         RowBox[{"Transpose", "[", 
          RowBox[{"{", 
           RowBox[{"samples", ",", "errdata"}], "}"}], "]"}], ",", " ", 
         RowBox[{"Frame", "\[Rule]", "True"}], ",", 
         RowBox[{
         "PlotLabel", "\[Rule]", 
          "\"\<Orthogonal error ||\!\(\*SuperscriptBox[\(Y\), \(T\)]\)Y - I\!\
\(\*SubscriptBox[\(||\), \(F\)]\) vs time\>\""}]}], "\[IndentingNewLine]", 
        "]"}]}]}], "\[IndentingNewLine]", "]"}]}], ";"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->627854489]
}, Open  ]],

Cell[CellGroupData[{

Cell["Square systems", "Subsection",
 CellID->1236503837],

Cell[TextData[{
 "This example concerns the solution of a matrix differential system on the \
orthogonal group ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["O", "3"], "(", "\[DoubleStruckCapitalR]", ")"}], 
   TraditionalForm]]],
 " (see ",
 "[",
 ButtonBox["Z98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#14383"],
 "]",
 ")."
}], "Text",
 CellID->946811583],

Cell["The matrix differential system is given by", "Text",
 CellID->1471284799],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SuperscriptBox["Y", "\[Prime]"], "=", 
     RowBox[{
      RowBox[{"F", "(", "Y", ")"}], " ", "Y"}]},
    {" ", "=", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"A", " ", "+", " ", 
        RowBox[{"(", 
         RowBox[{"I", "-", " ", 
          RowBox[{"Y", " ", 
           SuperscriptBox["Y", "T"]}]}], ")"}]}], ")"}], " ", "Y"}]}
   }], TraditionalForm]], "DisplayMath",
 CellID->1997329504],

Cell["with", "Text",
 CellID->706383500],

Cell[BoxData[
 FormBox[
  RowBox[{"A", " ", "=", " ", 
   TagBox[
    RowBox[{"(", "\[NoBreak]", GridBox[{
       {"0", 
        RowBox[{"-", "1"}], "1"},
       {"1", "0", "1"},
       {
        RowBox[{"-", "1"}], 
        RowBox[{"-", "1"}], "0"}
      }], "\[NoBreak]", ")"}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], TraditionalForm]], "DisplayMath",
 CellID->993735635],

Cell["and", "Text",
 CellID->2109052795],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["Y", "0"], "=", 
   RowBox[{
    SubscriptBox["I", "3"], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->939880022],

Cell["The solution evolves as:", "Text",
 CellID->639469266],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"Y", "(", "t", ")"}], " ", "=", " ", 
   TagBox[
    RowBox[{
     RowBox[{"exp", "[", 
      RowBox[{"t", " ", "A"}], "]"}], "."}],
    Function[BoxForm`e$, 
     MatrixForm[BoxForm`e$]]]}], TraditionalForm]], "DisplayMath",
 CellID->214526110],

Cell[TextData[{
 "The eigenvalues of Y(t) are ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Lambda]", "1"], "=", "1"}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Lambda]", "2"], "=", 
    RowBox[{"exp", "(", 
     RowBox[{"t", " ", "i", " ", 
      SqrtBox["3"]}], ")"}]}], TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Lambda]", "3"], "=", 
    RowBox[{"exp", "(", 
     RowBox[{
      RowBox[{"-", " ", "t"}], " ", "i", " ", 
      SqrtBox["3"]}], ")"}]}], TraditionalForm]]],
 ". Thus as ",
 StyleBox["t",
  FontSlant->"Italic"],
 " approaches ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Pi]", "/", 
    SqrtBox["3"]}], TraditionalForm]]],
 ", two of the eigenvalues of ",
 StyleBox["Y",
  FontSlant->"Italic"],
 "(",
 StyleBox["t",
  FontSlant->"Italic"],
 ") approach -1. The numerical integration is carried out on the interval [0, \
2]."
}], "Text",
 CellID->621082955],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"n", " ", "=", " ", "3"}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"A", " ", "=", " ", 
    TagBox[
     RowBox[{"(", "\[NoBreak]", GridBox[{
        {"0", 
         RowBox[{"-", "1"}], "1"},
        {"1", "0", "1"},
        {
         RowBox[{"-", "1"}], 
         RowBox[{"-", "1"}], "0"}
       }], "\[NoBreak]", ")"}],
     Function[BoxForm`e$, 
      MatrixForm[BoxForm`e$]]]}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Y", "=", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "[", 
        RowBox[{"i", ",", "j"}], "]"}], "[", "t", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "n"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"j", ",", "n"}], "}"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{"F", " ", "=", 
   RowBox[{"A", " ", "+", 
    RowBox[{"(", " ", 
     RowBox[{
      RowBox[{"IdentityMatrix", "[", "n", "]"}], " ", "-", 
      RowBox[{
       RowBox[{"Transpose", "[", "Y", "]"}], ".", "Y"}]}], ")"}]}]}], 
  ";"}]}], "Input",
 CellLabel->"In[7]:=",
 CellID->822056184],

Cell[BoxData[
 RowBox[{
  RowBox[{"(*", " ", 
   RowBox[{"Vector", " ", "differential", " ", "system"}], " ", "*)"}], 
  "\[IndentingNewLine]", "\n", 
  RowBox[{
   RowBox[{
    RowBox[{"system", "=", 
     RowBox[{"Thread", "[", 
      RowBox[{
       RowBox[{"Flatten", "[", 
        RowBox[{"D", "[", 
         RowBox[{"Y", ",", "t"}], "]"}], "]"}], "\[Equal]", 
       RowBox[{"Flatten", "[", 
        RowBox[{"F", ".", "Y"}], "]"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\[IndentingNewLine]", 
   RowBox[{"(*", " ", 
    RowBox[{"Vector", " ", "initial", " ", "conditions"}], " ", "*)"}], 
   "\[IndentingNewLine]", "\n", 
   RowBox[{
    RowBox[{"ics", "=", 
     RowBox[{"Thread", "[", 
      RowBox[{
       RowBox[{"Flatten", "[", 
        RowBox[{"(", 
         RowBox[{"Y", "/.", 
          RowBox[{"t", "\[Rule]", "0"}]}], ")"}], "]"}], "\[Equal]", 
       RowBox[{"Flatten", "[", 
        RowBox[{"IdentityMatrix", "[", 
         RowBox[{"Length", "[", "Y", "]"}], "]"}], "]"}]}], "]"}]}], ";"}], 
   "\[IndentingNewLine]", "\n", 
   RowBox[{
    RowBox[{"eqs", "=", 
     RowBox[{"{", 
      RowBox[{"system", ",", "ics"}], "}"}]}], ";"}], "\[IndentingNewLine]", 
   "\n", 
   RowBox[{
    RowBox[{"vars", " ", "=", " ", 
     RowBox[{"Flatten", "[", "Y", "]"}]}], ";"}], "\[IndentingNewLine]", "\n",
    
   RowBox[{
    RowBox[{"time", "=", 
     RowBox[{"{", 
      RowBox[{"t", ",", "0", ",", "2"}], "}"}]}], ";"}]}]}]], "Input",
 CellLabel->"In[8]:=",
 CellID->196207747],

Cell[TextData[{
 "This computes the solution using an explicit Runge-Kutta method. The \
appropriate initial step size and method order are selected automatically, \
and the step size may vary throughout the integration interval, which is \
chosen in order to satisfy local relative and absolute error tolerances. \
Alternatively, the order of the method could be specified by using a ",
 StyleBox["Method", "MR"],
 " option."
}], "Text",
 CellID->301146414],

Cell[BoxData[
 RowBox[{
  RowBox[{"solerk", "=", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"eqs", ",", "vars", ",", "time", ",", 
     RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->999996121],

Cell["\<\
This computes the orthogonal error, or absolute deviation from the orthogonal \
manifold, as the integration progresses. The error is of the order of the \
local accuracy of the numerical method.\
\>", "Text",
 CellID->1698026070],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"solerk", " ", "=", " ", 
    RowBox[{"Y", "/.", 
     RowBox[{"First", "[", "solerk", "]"}]}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{"OrthogonalErrorPlot", "[", "solerk", "]"}]}], "Input",
 CellLabel->"In[17]:=",
 CellID->1571727744],

Cell[BoxData[
 GraphicsBox[{{}, {}, {
    {Hue[0.67, 0.6, 0.6]}, 
    LineBox[{{0., 2.40012692342311*^-17}, {0.162169354870725, 
     8.2929064978211*^-11}, {0.362169354870725, 6.96303792362707*^-10}, {
     0.545081080619268, 7.5151457591622*^-10}, {0.735950240165429, 
     9.19644367978168*^-10}, {0.919892369024328, 9.20228180690687*^-10}, {
     1.10586512285269, 9.50078090781975*^-10}, {1.28975126052802, 
     9.40527714325108*^-10}, {1.47449330989041, 9.45894763421021*^-10}, {
     1.66468584216278, 1.03939751793139*^-9}, {1.83234292108139, 
     8.58094152945174*^-10}, {2., 7.28441570039912*^-10}}]}},
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  Frame->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotLabel->FormBox[
   "\"Orthogonal error ||\\!\\(Y\\^T\\)Y - I\\!\\(\\( || \\_F\\)\\) vs \
time\"", TraditionalForm],
  PlotRange->{{0., 2.}, {0., 1.03939751793139*^-9}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 230},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->1981098086]
}, Open  ]],

Cell["\<\
This computes the solution using an orthogonal projection method with an \
explicit Runge-Kutta method used for the basic integration step. The initial \
step size and method order are the same as earlier, but the step size \
sequence in the integration may differ.\
\>", "Text",
 CellID->995739125],

Cell[BoxData[
 RowBox[{
  RowBox[{"solop", "=", 
   RowBox[{"NDSolve", "[", 
    RowBox[{"eqs", ",", "vars", ",", "time", ",", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<OrthogonalProjection\>\"", ",", "\[IndentingNewLine]", 
        RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}], ",", " ",
         
        RowBox[{"Dimensions", "\[Rule]", 
         RowBox[{"Dimensions", "[", "Y", "]"}]}]}], "}"}]}]}], "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[19]:=",
 CellID->988158214],

Cell["\<\
Using the orthogonal projection method, the orthogonal error is reduced to \
approximately the level of roundoff in IEEE double-precision arithmetic.\
\>", "Text",
 CellID->1278047587],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"solop", " ", "=", " ", 
    RowBox[{"Y", "/.", 
     RowBox[{"First", "[", "solop", "]"}]}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{"OrthogonalErrorPlot", "[", "solop", "]"}]}], "Input",
 CellLabel->"In[20]:=",
 CellID->1353646550],

Cell[BoxData[
 GraphicsBox[{{}, {}, {
    {Hue[0.67, 0.6, 0.6]}, 
    LineBox[{{0., 4.58447480117194*^-17}, {0.162169354870725, 
     3.84874118183703*^-16}, {0.362169354870725, 1.97240424042898*^-16}, {
     0.545081080802435, 1.57009245868378*^-16}, {0.73595024058791, 
     1.84109660314757*^-16}, {0.919892369080456, 3.2840839953583*^-16}, {
     1.10586512305441, 5.57275697956085*^-17}, {1.28975126077468, 
     3.14669466688156*^-16}, {1.47449330884251, 3.24277311823604*^-16}, {
     1.66468584108403, 2.71947991102104*^-16}, {1.83234292054201, 
     2.77555756156289*^-16}, {2., 4.1725779438209*^-16}}]}},
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  Frame->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotLabel->FormBox[
   "\"Orthogonal error ||\\!\\(Y\\^T\\)Y - I\\!\\(\\( || \\_F\\)\\) vs \
time\"", TraditionalForm],
  PlotRange->{{0., 2.}, {4.58447480117194*^-17, 4.1725779438209*^-16}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 227},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->980833058]
}, Open  ]],

Cell["\<\
The Schulz iteration, using the incremental formulation, generally yields \
smaller errors than the direct singular value decomposition.\
\>", "Text",
 CellID->1605354731],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.354926 0.00413632 0.0147151 0.00443884 [
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
0 g
.25 Mabswid
[ ] 0 Mabsdash
.25537 .11267 m
.25537 .11762 L
s
[(0)] .24141 .07724 -1 -1 Mshowa
.43117 .11267 m
.43117 .11762 L
s
[(0.5)] .39239 .07724 -1 -1 Mshowa
.60699 .11267 m
.60699 .11762 L
s
[(1)] .59122 .07724 -1 -1 Mshowa
.7828 .11267 m
.7828 .11762 L
s
[(1.5)] .74312 .07724 -1 -1 Mshowa
.95861 .11267 m
.95861 .11762 L
s
[(2)] .94465 .07724 -1 -1 Mshowa
.125 Mabswid
.29053 .11267 m
.29053 .11564 L
s
.32569 .11267 m
.32569 .11564 L
s
.36086 .11267 m
.36086 .11564 L
s
.39602 .11267 m
.39602 .11564 L
s
.46634 .11267 m
.46634 .11564 L
s
.5015 .11267 m
.5015 .11564 L
s
.53666 .11267 m
.53666 .11564 L
s
.57182 .11267 m
.57182 .11564 L
s
.64215 .11267 m
.64215 .11564 L
s
.67731 .11267 m
.67731 .11564 L
s
.71248 .11267 m
.71248 .11564 L
s
.74764 .11267 m
.74764 .11564 L
s
.81796 .11267 m
.81796 .11564 L
s
.85312 .11267 m
.85312 .11564 L
s
.88828 .11267 m
.88828 .11564 L
s
.92344 .11267 m
.92344 .11564 L
s
.25 Mabswid
.23779 .11267 m
.97619 .11267 L
s
.23779 .12433 m
.2424 .12433 L
s
[(0)] .20684 .11185 -1 -1 Mshowa
.23779 .20342 m
.2424 .20342 L
s
[(2)] .07344 .18456 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .18456 -1 -1 Mshowa
P
[(10)] .12308 .18456 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .20786 -1 -1 Mshowa
P
.23779 .28251 m
.2424 .28251 L
s
[(4)] .07344 .26365 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .26365 -1 -1 Mshowa
P
[(10)] .12308 .26365 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .28695 -1 -1 Mshowa
P
.23779 .3616 m
.2424 .3616 L
s
[(6)] .07344 .34274 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .34274 -1 -1 Mshowa
P
[(10)] .12308 .34274 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .36604 -1 -1 Mshowa
P
.23779 .4407 m
.2424 .4407 L
s
[(8)] .07344 .42183 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .42183 -1 -1 Mshowa
P
[(10)] .12308 .42183 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .44514 -1 -1 Mshowa
P
.23779 .51979 m
.2424 .51979 L
s
[(1)] .07344 .50092 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .50092 -1 -1 Mshowa
P
[(10)] .12308 .50092 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-15)] .17272 .52423 -1 -1 Mshowa
P
.23779 .59888 m
.2424 .59888 L
s
[(1.2)] .02381 .58001 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .58001 -1 -1 Mshowa
P
[(10)] .12308 .58001 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-15)] .17272 .60332 -1 -1 Mshowa
P
.125 Mabswid
.23779 .1441 m
.24055 .1441 L
s
.23779 .16387 m
.24055 .16387 L
s
.23779 .18364 m
.24055 .18364 L
s
.23779 .22319 m
.24055 .22319 L
s
.23779 .24296 m
.24055 .24296 L
s
.23779 .26274 m
.24055 .26274 L
s
.23779 .30228 m
.24055 .30228 L
s
.23779 .32206 m
.24055 .32206 L
s
.23779 .34183 m
.24055 .34183 L
s
.23779 .38138 m
.24055 .38138 L
s
.23779 .40115 m
.24055 .40115 L
s
.23779 .42092 m
.24055 .42092 L
s
.23779 .46047 m
.24055 .46047 L
s
.23779 .48024 m
.24055 .48024 L
s
.23779 .50002 m
.24055 .50002 L
s
.23779 .53956 m
.24055 .53956 L
s
.23779 .55934 m
.24055 .55934 L
s
.23779 .57911 m
.24055 .57911 L
s
.25 Mabswid
.23779 .11267 m
.23779 .6024 L
s
.25537 .59745 m
.25537 .6024 L
s
.43117 .59745 m
.43117 .6024 L
s
.60699 .59745 m
.60699 .6024 L
s
.7828 .59745 m
.7828 .6024 L
s
.95861 .59745 m
.95861 .6024 L
s
.125 Mabswid
.29053 .59943 m
.29053 .6024 L
s
.32569 .59943 m
.32569 .6024 L
s
.36086 .59943 m
.36086 .6024 L
s
.39602 .59943 m
.39602 .6024 L
s
.46634 .59943 m
.46634 .6024 L
s
.5015 .59943 m
.5015 .6024 L
s
.53666 .59943 m
.53666 .6024 L
s
.57182 .59943 m
.57182 .6024 L
s
.64215 .59943 m
.64215 .6024 L
s
.67731 .59943 m
.67731 .6024 L
s
.71248 .59943 m
.71248 .6024 L
s
.74764 .59943 m
.74764 .6024 L
s
.81796 .59943 m
.81796 .6024 L
s
.85312 .59943 m
.85312 .6024 L
s
.88828 .59943 m
.88828 .6024 L
s
.92344 .59943 m
.92344 .6024 L
s
.25 Mabswid
.23779 .6024 m
.97619 .6024 L
s
.97157 .12433 m
.97619 .12433 L
s
.97157 .20342 m
.97619 .20342 L
s
.97157 .28251 m
.97619 .28251 L
s
.97157 .3616 m
.97619 .3616 L
s
.97157 .4407 m
.97619 .4407 L
s
.97157 .51979 m
.97619 .51979 L
s
.97157 .59888 m
.97619 .59888 L
s
.125 Mabswid
.97342 .1441 m
.97619 .1441 L
s
.97342 .16387 m
.97619 .16387 L
s
.97342 .18364 m
.97619 .18364 L
s
.97342 .22319 m
.97619 .22319 L
s
.97342 .24296 m
.97619 .24296 L
s
.97342 .26274 m
.97619 .26274 L
s
.97342 .30228 m
.97619 .30228 L
s
.97342 .32206 m
.97619 .32206 L
s
.97342 .34183 m
.97619 .34183 L
s
.97342 .38138 m
.97619 .38138 L
s
.97342 .40115 m
.97619 .40115 L
s
.97342 .42092 m
.97619 .42092 L
s
.97342 .46047 m
.97619 .46047 L
s
.97342 .48024 m
.97619 .48024 L
s
.97342 .50002 m
.97619 .50002 L
s
.97342 .53956 m
.97619 .53956 L
s
.97342 .55934 m
.97619 .55934 L
s
.97342 .57911 m
.97619 .57911 L
s
.25 Mabswid
.97619 .11267 m
.97619 .6024 L
s
.5 Mabswid
[ 2.23146 2.23146 ] 0 Mabsdash
.25537 .13124 m
.29051 .44771 L
.32447 .32036 L
.36663 .46844 L
.41088 .41048 L
.45989 .24733 L
.5117 .4797 L
.56707 .43942 L
.62368 .59074 L
.68138 .23492 L
.73975 .46833 L
.79965 .51616 L
.86096 .39353 L
.90979 .39992 L
.95861 .29377 L
s
[ ] 0 Mabsdash
.25537 .13214 m
.29051 .16833 L
.32447 .23476 L
.36663 .23652 L
.41088 .16625 L
.45989 .24367 L
.5117 .24991 L
.56707 .24356 L
.62368 .22336 L
.68138 .13713 L
.73975 .32243 L
.79965 .17756 L
.86096 .27463 L
.90979 .25093 L
.95861 .2575 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
 ImageSize->{288, 177.938},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"From In[3]:=",
 CellID->1679290373,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"],
 ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {85.5074, -3.31577, 0.844462, \
0.786909}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Rectangular systems", "Subsection",
 CellID->1370560260],

Cell[TextData[{
 "In the following example it is shown how the implementation of the \
orthogonal projection method also works for rectangular matrix differential \
systems. Formally stated, we are interested in solving ordinary differential \
equations on the ",
 StyleBox["Stiefel manifold",
  FontSlant->"Italic"],
 ", the set of n\[Times]p orthogonal matrices with p < n."
}], "Text",
 CellID->1168948121],

Cell[TextData[{
 StyleBox["Definition",
  FontWeight->"Bold"],
 " The ",
 StyleBox["Stiefel manifold",
  FontSlant->"Italic"],
 " of n\[Times]p orthogonal matrices is the set ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["V", 
     RowBox[{"n", ",", "p"}]], "(", "\[DoubleStruckCapitalR]", ")"}], 
   TraditionalForm]]],
 " = {Y\[Element]",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", 
    RowBox[{"n", "\[Times]", "p"}]], TraditionalForm]]],
 " | ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["Y", "T"], " ", "Y"}], " ", "=", " ", 
    SubscriptBox["I", "p"]}], TraditionalForm]]],
 "}, 1\[LessEqual]p<n, where ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", "p"], TraditionalForm]]],
 "is the p\[Times]p identity matrix."
}], "Text",
 CellID->1996309344],

Cell["\<\
Solutions that evolve on the Stiefel manifold find numerous applications such \
as eigenvalue problems in numerical linear algebra, computation of Lyapunov \
exponents for dynamical systems and signal processing.\
\>", "Text",
 CellID->1644957818],

Cell[TextData[{
 "Consider an example adapted from [",
 ButtonBox["DL01",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2703"],
 "]:"
}], "Text",
 CellID->411993719],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SuperscriptBox["q", "\[Prime]"], "(", "t", ")"}], "=", 
    RowBox[{"A", " ", 
     RowBox[{"q", "(", "t", ")"}]}]}], ",", 
   RowBox[{"t", ">", "0"}], ",", 
   RowBox[{
    RowBox[{"q", "(", "0", ")"}], "=", 
    FormBox[
     SubscriptBox["q", "0"],
     TraditionalForm]}]}], TraditionalForm]], "DisplayMath",
 CellID->1104654476],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["q", "0"], " ", "=", " ", 
    RowBox[{"1", "/", 
     SuperscriptBox[
      RowBox[{
       SqrtBox["n"], " ", "[", 
       RowBox[{"1", ",", "\[Ellipsis]", ",", "1"}], "]"}], "T"]}]}], 
   TraditionalForm]]],
 " , A = ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"diag", "[", 
     RowBox[{
      SubscriptBox["a", "1"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["a", "n"]}], "]"}], "\[Element]", 
    SuperscriptBox["\[DoubleStruckCapitalR]", 
     RowBox[{"n", "\[Times]", "n"}]]}], TraditionalForm]]],
 ", with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["a", "i"], "=", " ", 
     RowBox[{
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"-", "1"}], ")"}], "i"], "\[Alpha]"}]}], " ", ",", " ", 
    RowBox[{"i", "=", "1"}], ",", "\[Ellipsis]", ",", "n"}], 
   TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Alpha]", ">", "0"}], TraditionalForm]]],
 "."
}], "Text",
 CellID->125328659],

Cell["The exact solution is given by:", "Text",
 CellID->1648497557],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"q", "(", "t", ")"}], " ", "=", " ", 
   RowBox[{
    FractionBox["1", 
     SqrtBox["n"]], 
    RowBox[{
     TagBox[
      RowBox[{"(", "\[NoBreak]", GridBox[{
         {
          RowBox[{"exp", "(", 
           RowBox[{
            SubscriptBox["a", "1"], "t"}], ")"}]},
         {"\[VerticalEllipsis]"},
         {
          RowBox[{"exp", "(", 
           RowBox[{
            SubscriptBox["a", "n"], "t"}], ")"}]}
        }], "\[NoBreak]", ")"}],
      Function[BoxForm`e$, 
       MatrixForm[BoxForm`e$]]], "."}]}]}], TraditionalForm]], "DisplayMath",
 CellID->2051466059],

Cell["Normalizing q(t) as:", "Text",
 CellID->1596613459],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"Y", "(", "t", ")"}], " ", "=", " ", 
   RowBox[{
    FractionBox[
     RowBox[{"q", "(", "t", ")"}], 
     RowBox[{"||", 
      RowBox[{"q", "(", "t", ")"}], "||"}]], 
    TagBox[
     RowBox[{"\[Element]", 
      SuperscriptBox["\[DoubleStruckCapitalR]", 
       RowBox[{"n", "\[Times]", "1"}]]}],
     Function[BoxForm`e$, 
      MatrixForm[BoxForm`e$]]]}]}], TraditionalForm]], "DisplayMath",
 CellID->1498343413],

Cell[TextData[{
 "it follows that Y(t) satisfies the following weak skew-symmetric system on \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["V", 
     RowBox[{"n", ",", " ", "1"}]], "(", "\[DoubleStruckCapitalR]", ")"}], 
   TraditionalForm]]],
 ":"
}], "Text",
 CellID->610365912],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SuperscriptBox["Y", "\[Prime]"], "=", 
     RowBox[{
      RowBox[{"F", "(", "Y", ")"}], " ", "Y"}]},
    {" ", "=", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        SubscriptBox["I", "n"], "-", " ", 
        RowBox[{"Y", " ", 
         SuperscriptBox["Y", "T"]}]}], ")"}], " ", "A", " ", "Y"}]}
   }], TraditionalForm]], "DisplayMath",
 CellID->442865446],

Cell["\<\
In the following example, the system is solved on the interval [0, 5] with \
\[Alpha] = 9/10 and dimension n = 2.\
\>", "Text",
 CellID->1459912560],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"p", " ", "=", " ", "1"}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"n", " ", "=", " ", "2"}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"\[Alpha]", " ", "=", " ", 
    FractionBox["9", "10"]}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"ics", " ", "=", " ", 
    RowBox[{
     FractionBox["1", 
      SqrtBox["n"]], 
     RowBox[{"Table", "[", 
      RowBox[{"1", ",", 
       RowBox[{"{", "n", "}"}]}], "]"}]}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"avec", " ", "=", " ", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], "i"], " ", "\[Alpha]"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", " ", "n"}], "}"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"A", " ", "=", " ", 
    RowBox[{"DiagonalMatrix", "[", "avec", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"Y", "=", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"y", "[", 
        RowBox[{"i", ",", "1"}], "]"}], "[", "t", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", " ", "n"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"j", ",", "p"}], "}"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"F", " ", "=", " ", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"IdentityMatrix", "[", 
        RowBox[{"Length", "[", "Y", "]"}], "]"}], " ", "-", 
       RowBox[{"Y", ".", 
        RowBox[{"Transpose", "[", "Y", "]"}]}]}], ")"}], ".", "A"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"system", "=", 
    RowBox[{"Thread", "[", 
     RowBox[{
      RowBox[{"Flatten", "[", 
       RowBox[{"D", "[", 
        RowBox[{"Y", ",", "t"}], "]"}], "]"}], "\[Equal]", 
      RowBox[{"Flatten", "[", 
       RowBox[{"F", ".", "Y"}], "]"}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"ics", "=", 
    RowBox[{"Thread", "[", 
     RowBox[{
      RowBox[{"Flatten", "[", 
       RowBox[{"(", 
        RowBox[{"Y", "/.", 
         RowBox[{"t", "\[Rule]", "0"}]}], ")"}], "]"}], "\[Equal]", "ics"}], 
     "]"}]}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"eqs", "=", 
    RowBox[{"{", 
     RowBox[{"system", ",", "ics"}], "}"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"vars", " ", "=", " ", 
    RowBox[{"Flatten", "[", "Y", "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"tfinal", " ", "=", " ", "5."}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{"time", "=", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "tfinal"}], "}"}]}], ";"}]}], "Input",
 CellLabel->"In[22]:=",
 CellID->1550245934],

Cell["\<\
This computes the exact solution which can be evaluated throughout the \
integration interval.\
\>", "Text",
 CellID->1082334712],

Cell[BoxData[
 RowBox[{
  RowBox[{"solexact", " ", "=", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"Transpose", "[", 
      RowBox[{"{", 
       RowBox[{"(", 
        FractionBox["#", 
         RowBox[{"Norm", "[", 
          RowBox[{"#", ",", " ", "2"}], "]"}]], ")"}], "}"}], "]"}], "&"}], 
    " ", "@", " ", 
    RowBox[{"(", 
     FractionBox[
      RowBox[{"Exp", "[", 
       RowBox[{"avec", " ", "t"}], "]"}], 
      SqrtBox["n"]], " ", ")"}]}]}], ";"}]], "Input",
 CellLabel->"In[36]:=",
 CellID->1674438670],

Cell["\<\
This computes the solution using an explicit Runge-Kutta method.\
\>", "Text",
 CellID->23845241],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"solerk", "=", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"eqs", ",", "vars", ",", "time", ",", 
      RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}]}], "]"}]}], 
   ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{"solerk", " ", "=", " ", 
   RowBox[{"Y", " ", "/.", " ", 
    RowBox[{"First", "[", "solerk", "]"}]}]}], ";"}]}], "Input",
 CellLabel->"In[37]:=",
 CellID->1904390897],

Cell["\<\
This computes the componentwise absolute global error at the end of the \
integration interval.\
\>", "Text",
 CellID->1975585085],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"solexact", " ", "-", "solerk"}], ")"}], " ", "/.", " ", 
  RowBox[{"t", "\[Rule]", "tfinal"}]}]], "Input",
 CellLabel->"In[39]:=",
 CellID->1023841362],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"-", "2.0340662016055458`*^-11"}], "}"}], ",", 
   RowBox[{"{", "2.963185252724543`*^-13", "}"}]}], "}"}]], "Output",
 ImageSize->{245, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->1562479585]
}, Open  ]],

Cell["\<\
This computes the orthogonal error - a measure of the deviation from the \
Stiefel manifold.\
\>", "Text",
 CellID->1399829181],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"OrthogonalErrorPlot", "[", "solerk", "]"}]], "Input",
 CellLabel->"In[40]:=",
 CellID->1148779652],

Cell[BoxData[
 GraphicsBox[{{}, {}, {
    {Hue[0.67, 0.6, 0.6]}, 
    LineBox[{{0., 2.22044604925031*^-16}, {0.180616609516074, 
     6.5200178589464*^-11}, {0.302975391734647, 6.74683210062938*^-10}}], 
    LineBox[{{1.10134981664172, 6.74683210062938*^-10}, {1.28480070946813, 
     4.55806947741166*^-10}, {1.51182582305875, 2.6656366003408*^-10}, {
     1.75976325844634, 1.23952181851905*^-10}, {2.01669306275034, 
     4.87450080299823*^-11}, {2.29538982454181, 3.28892468814956*^-12}, {
     2.5807074466099, 1.02320374395504*^-11}, {2.8827684307887, 
     1.3500089934837*^-11}, {3.2001332986822, 1.17075238392772*^-11}, {
     3.53688383012484, 8.66984262160031*^-12}, {3.89480355180645, 
     5.75950398484792*^-12}, {4.2774830806186, 3.5154101851731*^-12}, {
     4.6387415403093, 1.92645899232957*^-12}, {5., 1.03028696685215*^-12}}]}},
  
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  Frame->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotLabel->FormBox[
   "\"Orthogonal error ||\\!\\(Y\\^T\\)Y - I\\!\\(\\( || \\_F\\)\\) vs \
time\"", TraditionalForm],
  PlotRange->{{0., 5.}, {0., 6.74683210062938*^-10}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 230},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[40]=",
 CellID->1655537558]
}, Open  ]],

Cell["\<\
This computes the solution using an orthogonal projection method with an \
explicit Runge-Kutta method as the basic numerical integration scheme.\
\>", "Text",
 CellID->240503747],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"solop", "=", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"eqs", ",", "vars", ",", "time", ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<OrthogonalProjection\>\"", ",", "\[IndentingNewLine]", 
         RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}], ",", 
         " ", 
         RowBox[{"Dimensions", "\[Rule]", 
          RowBox[{"Dimensions", "[", "Y", "]"}]}]}], "}"}]}]}], "]"}]}], 
   ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{
  RowBox[{"solop", "=", " ", 
   RowBox[{"Y", " ", "/.", " ", 
    RowBox[{"First", "[", "solop", "]"}]}]}], ";"}]}], "Input",
 CellLabel->"In[41]:=",
 CellID->279343591],

Cell["\<\
The componentwise absolute global error at the end of the integration \
interval is roughly the same as before since the absolute and relative \
tolerances used in the numerical integration are the same.\
\>", "Text",
 CellID->1656835346],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{"solexact", " ", "-", " ", "solop"}], ")"}], " ", "/.", " ", 
  RowBox[{"t", "\[Rule]", "tfinal"}]}]], "Input",
 CellLabel->"In[43]:=",
 CellID->667476263],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"-", "2.0340697713411987`*^-11"}], "}"}], ",", 
   RowBox[{"{", "2.55351295663786`*^-15", "}"}]}], "}"}]], "Output",
 ImageSize->{245, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[43]=",
 CellID->1171477638]
}, Open  ]],

Cell["\<\
Using the orthogonal projection method however, the deviation from the \
Stiefel manifold is reduced to the level of roundoff.\
\>", "Text",
 CellID->1188642071],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"OrthogonalErrorPlot", "[", "solop", "]"}]], "Input",
 CellLabel->"In[44]:=",
 CellID->2038036524],

Cell[BoxData[
 GraphicsBox[{{}, {}, {
    {Hue[0.67, 0.6, 0.6]}, 
    LineBox[{{0., 2.22044604925031*^-16}, {0.180616609516074, 0.}, {
     0.433096035393688, 0.}, {0.63565985039076, 2.22044604925031*^-16}, {
     0.856601410363727, 1.11022302462516*^-16}, {1.06817691203794, 0.}, {
     1.28480070852058, 0.}, {1.51182582166789, 0.}, {1.75976325834322, 
     2.22044604925031*^-16}, {2.01669305937616, 2.22044604925031*^-16}, {
     2.29538981996553, 0.}, {2.5807074412028, 2.22044604925031*^-16}, {
     2.88276842457006, 2.22044604925031*^-16}, {3.20013329156123, 
     2.22044604925031*^-16}, {3.53688382240843, 1.11022302462516*^-16}, {
     3.8948035433052, 0.}, {4.27748307117008, 2.22044604925031*^-16}, {
     4.63874153558504, 0.}, {5., 0.}}]}},
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  Frame->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotLabel->FormBox[
   "\"Orthogonal error ||\\!\\(Y\\^T\\)Y - I\\!\\(\\( || \\_F\\)\\) vs \
time\"", TraditionalForm],
  PlotRange->{{0., 5.}, {0., 2.22044604925031*^-16}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 227},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[44]=",
 CellID->1847780704]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Implementation", "Section",
 CellTags->"s:3",
 CellID->1037467767],

Cell[TextData[{
 "The implementation of the method ",
 StyleBox["OrthogonalProjection", "MR"],
 " has three basic components:"
}], "Text",
 CellID->1535510694],

Cell[TextData[{
 "Initialization. Set up the base method to use in the integration, \
determining any method coefficients and setting up any workspaces that should \
be used. This is done once, before any actual integration is carried out, and \
the resulting ",
 StyleBox["MethodData", "MR"],
 " object is validated so that it does not need to be checked at each \
integration step. At this stage the system dimensions and initial conditions \
are checked for consistency."
}], "BulletedText",
 CellID->1318156515],

Cell["Invoke the base numerical integration method at each step.", \
"BulletedText",
 CellID->2142122243],

Cell[CellGroupData[{

Cell["\<\
Perform an orthogonal projection. This performs various tests such as \
checking that the basic integration proceeded correctly and that the Schulz \
iteration converges.\
\>", "BulletedText",
 CellID->1660839353],

Cell[TextData[{
 "Options can be used to modify the stopping criteria for the Schulz \
iteration. One option provided by our code is ",
 StyleBox["IterationSafetyFactor ", "MR"],
 "which allows control over the tolerance \[Tau] of the iteration. The factor \
is combined with a Unit in the Last Place, determined according to the \
working precision used in the integration (",
 Cell[BoxData[
  FormBox[
   RowBox[{"ULP", "\[TildeTilde]", 
    RowBox[{"2.22045", "\[Cross]", 
     SuperscriptBox["10", 
      RowBox[{"-", "16"}]]}]}], TraditionalForm]]],
 " for IEEE double precision). "
}], "Text",
 CellID->819170424],

Cell[TextData[{
 "The Frobenius norm used for the stopping criterion can be computed \
efficiently using the LAPACK LANGE functions [",
 ButtonBox["LAPACK99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#18138"],
 "]"
}], "Text",
 CellID->2046104655],

Cell[TextData[{
 "The option ",
 StyleBox["MaxIterations", "MR"],
 " controls the maximum number of iterations ",
 Cell[BoxData[
  FormBox["imax", TraditionalForm]]],
 " that should be carried out."
}], "Text",
 CellID->1109969164]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
 CellTags->"s:4",
 CellID->170030189],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Dimensions",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Dimensions"], 
    RowBox[{"{", "}"}], Cell[
    "Specifies the dimensions of the matrix differential system", 
     "TableText"]},
   {"IterationSafetyFactor", 
    FractionBox[
     StyleBox["1", "InlineCode"], "10"], Cell[TextData[{
     "Specifies the safety factor to use in the termination criterion for the \
Schulz iteration (",
     
     CounterBox["NumberedEquation", "SchulzIteration"],
     ")."
    }], "TableText"]},
   {
    ButtonBox["MaxIterations",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxIterations"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "Specifies the maximum number of iterations to use in the Schulz \
iteration (",
     
     CounterBox["NumberedEquation", "SchulzIteration"],
     ")."
    }], "TableText"]},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], "StiffnessSwitching", Cell[
    "Specifies the method to use for the numerical integration.", 
     "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->508986919],

Cell[TextData[{
 "Options of the method ",
 Cell[BoxData["OrthogonalProjection"], "InlineFormula"],
 ". "
}], "Caption",
 CellID->1552834611]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"OrthogonalProjection Method for NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveFixedStep"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveProjection"]}], "Text",
       FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 56.4934032}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "Consider the matrix differential equation: where the initial value y_0= \
y(0)\\[Element] \\[DoubleStruckCapitalR]^mxp is given. Assume that y_0^Ty_0 = \
I, that the solution has the property of preserving orthonormality, y(t)^T \
y(t) = I, and that it has full rank for all t >= 0. From a numerical \
perspective, a key issue is how to numerically integrate an orthogonal matrix \
differential system in such a way that the numerical solution remains \
orthogonal. There are several strategies that are possible. One approach, \
suggested in [DRV94], is to use an implicit Runge-Kutta method (such as the \
Gauss scheme). Some alternative strategies are described in [DV99] and \
[DL01].", "synonyms" -> {}, "title" -> 
    "OrthogonalProjection Method for NDSolve", "type" -> "Tutorial", "uri" -> 
    "tutorial/NDSolveOrthogonalProjection"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:1"->{
  Cell[728, 29, 94, 2, 70, "Title",
   CellTags->"c:1",
   CellID->569226817]},
 "s:1"->{
  Cell[847, 35, 69, 2, 70, "Section",
   CellTags->"s:1",
   CellID->293332788]},
 "SchulzIteration"->{
  Cell[9710, 364, 547, 19, 70, "NumberedEquation",
   CellTags->"SchulzIteration",
   CellID->1828686581]},
 "s:2"->{
  Cell[17303, 645, 65, 2, 70, "Section",
   CellTags->"s:2",
   CellID->327307144]},
 "s:3"->{
  Cell[60121, 2265, 72, 2, 70, "Section",
   CellTags->"s:3",
   CellID->1037467767]},
 "s:4"->{
  Cell[62412, 2342, 71, 2, 70, "Section",
   CellTags->"s:4",
   CellID->170030189]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:1", 66875, 2460},
 {"s:1", 66965, 2464},
 {"SchulzIteration", 67069, 2468},
 {"s:2", 67187, 2472},
 {"s:3", 67282, 2476},
 {"s:4", 67379, 2480}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 103, 2, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[728, 29, 94, 2, 70, "Title",
 CellTags->"c:1",
 CellID->569226817],
Cell[CellGroupData[{
Cell[847, 35, 69, 2, 70, "Section",
 CellTags->"s:1",
 CellID->293332788],
Cell[919, 39, 79, 1, 70, "Text",
 CellID->2028167128],
Cell[1001, 42, 310, 10, 70, "DisplayMath",
 CellID->1060194370],
Cell[1314, 54, 924, 32, 70, "Text",
 CellID->22383832],
Cell[2241, 88, 747, 19, 70, "Text",
 CellID->1321681495],
Cell[2991, 109, 230, 5, 70, "Text",
 CellID->2115220046],
Cell[3224, 116, 480, 10, 70, "Text",
 CellID->1504773441],
Cell[CellGroupData[{
Cell[3729, 130, 58, 1, 70, "Subsection",
 CellID->1383769894],
Cell[3790, 133, 489, 14, 70, "Text",
 CellID->2138858796],
Cell[4282, 149, 71, 1, 70, "BulletedText",
 CellID->1478293461],
Cell[4356, 152, 61, 1, 70, "BulletedText",
 CellID->1826935855],
Cell[CellGroupData[{
Cell[4442, 157, 72, 1, 70, "BulletedText",
 CellID->938809808],
Cell[4517, 160, 360, 7, 70, "Text",
 CellID->353456932],
Cell[4880, 169, 192, 4, 70, "Text",
 CellID->1993458239],
Cell[5075, 175, 1982, 73, 70, "Text",
 CellID->1838466452],
Cell[7060, 250, 1093, 43, 70, "Text",
 CellID->843430310],
Cell[8156, 295, 106, 3, 70, "Text",
 CellID->760825666],
Cell[8265, 300, 533, 19, 70, "DisplayMath",
 CellID->1907055565],
Cell[8801, 321, 187, 7, 70, "Text",
 CellID->1669934484]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9037, 334, 58, 1, 70, "Subsection",
 CellID->111723291],
Cell[9098, 337, 396, 16, 70, "Text",
 CellID->233496521],
Cell[9497, 355, 138, 4, 70, "Text",
 CellID->1793217278],
Cell[9638, 361, 69, 1, 70, "Text",
 CellID->252956962],
Cell[9710, 364, 547, 19, 70, "NumberedEquation",
 CellTags->"SchulzIteration",
 CellID->1828686581],
Cell[10260, 385, 481, 15, 70, "Text",
 CellID->182586918],
Cell[10744, 402, 1301, 36, 70, "Text",
 CellID->86620304]
}, Open  ]],
Cell[CellGroupData[{
Cell[12082, 443, 62, 1, 70, "Subsection",
 CellID->378913166],
Cell[12147, 446, 595, 17, 70, "Text",
 CellID->875969885],
Cell[12745, 465, 92, 1, 70, "Text",
 CellID->1270581329],
Cell[12840, 468, 1348, 56, 70, "Text",
 CellID->1272425816]
}, Open  ]],
Cell[CellGroupData[{
Cell[14225, 529, 64, 1, 70, "Subsection",
 CellID->1445196703],
Cell[14292, 532, 762, 24, 70, "Text",
 CellID->1563914118],
Cell[15057, 558, 262, 8, 70, "Text",
 CellID->1153109297],
Cell[15322, 568, 1628, 62, 70, "Text",
 CellID->357907425],
Cell[16953, 632, 301, 7, 70, "Text",
 CellID->1586297950]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[17303, 645, 65, 2, 70, "Section",
 CellTags->"s:2",
 CellID->327307144],
Cell[CellGroupData[{
Cell[17393, 651, 70, 1, 70, "Subsection",
 CellID->325643823],
Cell[17466, 654, 337, 13, 70, "Text",
 CellID->943587744],
Cell[17806, 669, 260, 8, 70, "Input",
 CellID->1711077038],
Cell[18069, 679, 297, 10, 70, "Text",
 CellID->317018971],
Cell[18369, 691, 457, 14, 70, "Input",
 CellID->274397557],
Cell[18829, 707, 149, 4, 70, "Text",
 CellID->1390529246],
Cell[18981, 713, 1200, 33, 70, "Input",
 CellID->1255828779],
Cell[20184, 748, 2393, 58, 70, "Input",
 CellID->627854489]
}, Open  ]],
Cell[CellGroupData[{
Cell[22614, 811, 57, 1, 70, "Subsection",
 CellID->1236503837],
Cell[22674, 814, 400, 16, 70, "Text",
 CellID->946811583],
Cell[23077, 832, 79, 1, 70, "Text",
 CellID->1471284799],
Cell[23159, 835, 446, 15, 70, "DisplayMath",
 CellID->1997329504],
Cell[23608, 852, 40, 1, 70, "Text",
 CellID->706383500],
Cell[23651, 855, 393, 14, 70, "DisplayMath",
 CellID->993735635],
Cell[24047, 871, 40, 1, 70, "Text",
 CellID->2109052795],
Cell[24090, 874, 170, 6, 70, "DisplayMath",
 CellID->939880022],
Cell[24263, 882, 60, 1, 70, "Text",
 CellID->639469266],
Cell[24326, 885, 291, 10, 70, "DisplayMath",
 CellID->214526110],
Cell[24620, 897, 972, 40, 70, "Text",
 CellID->621082955],
Cell[25595, 939, 1174, 41, 70, "Input",
 CellID->822056184],
Cell[26772, 982, 1498, 45, 70, "Input",
 CellID->196207747],
Cell[28273, 1029, 458, 9, 70, "Text",
 CellID->301146414],
Cell[28734, 1040, 269, 8, 70, "Input",
 CellID->999996121],
Cell[29006, 1050, 240, 5, 70, "Text",
 CellID->1698026070],
Cell[CellGroupData[{
Cell[29271, 1059, 296, 9, 70, "Input",
 CellID->1571727744],
Cell[29570, 1070, 1128, 28, 251, "Output",
 CellID->1981098086]
}, Open  ]],
Cell[30713, 1101, 309, 6, 70, "Text",
 CellID->995739125],
Cell[31025, 1109, 525, 14, 70, "Input",
 CellID->988158214],
Cell[31553, 1125, 194, 4, 70, "Text",
 CellID->1278047587],
Cell[CellGroupData[{
Cell[31772, 1133, 293, 9, 70, "Input",
 CellID->1353646550],
Cell[32068, 1144, 1146, 28, 248, "Output",
 CellID->980833058]
}, Open  ]],
Cell[33229, 1175, 181, 4, 70, "Text",
 CellID->1605354731],
Cell[33413, 1181, 11413, 541, 70, 6738, 460, "GraphicsData", "PostScript", \
"Graphics",
 CellID->1679290373]
}, Open  ]],
Cell[CellGroupData[{
Cell[44863, 1727, 62, 1, 70, "Subsection",
 CellID->1370560260],
Cell[44928, 1730, 409, 9, 70, "Text",
 CellID->1168948121],
Cell[45340, 1741, 825, 31, 70, "Text",
 CellID->1996309344],
Cell[46168, 1774, 257, 5, 70, "Text",
 CellID->1644957818],
Cell[46428, 1781, 189, 7, 70, "Text",
 CellID->411993719],
Cell[46620, 1790, 398, 14, 70, "DisplayMath",
 CellID->1104654476],
Cell[47021, 1806, 1045, 40, 70, "Text",
 CellID->125328659],
Cell[48069, 1848, 68, 1, 70, "Text",
 CellID->1648497557],
Cell[48140, 1851, 626, 22, 70, "DisplayMath",
 CellID->2051466059],
Cell[48769, 1875, 57, 1, 70, "Text",
 CellID->1596613459],
Cell[48829, 1878, 463, 15, 70, "DisplayMath",
 CellID->1498343413],
Cell[49295, 1895, 294, 11, 70, "Text",
 CellID->610365912],
Cell[49592, 1908, 410, 14, 70, "DisplayMath",
 CellID->442865446],
Cell[50005, 1924, 158, 4, 70, "Text",
 CellID->1459912560],
Cell[50166, 1930, 2982, 103, 70, "Input",
 CellID->1550245934],
Cell[53151, 2035, 139, 4, 70, "Text",
 CellID->1082334712],
Cell[53293, 2041, 518, 18, 70, "Input",
 CellID->1674438670],
Cell[53814, 2061, 107, 3, 70, "Text",
 CellID->23845241],
Cell[53924, 2066, 453, 13, 70, "Input",
 CellID->1904390897],
Cell[54380, 2081, 140, 4, 70, "Text",
 CellID->1975585085],
Cell[CellGroupData[{
Cell[54545, 2089, 203, 6, 70, "Input",
 CellID->1023841362],
Cell[54751, 2097, 319, 10, 39, "Output",
 CellID->1562479585]
}, Open  ]],
Cell[55085, 2110, 137, 4, 70, "Text",
 CellID->1399829181],
Cell[CellGroupData[{
Cell[55247, 2118, 122, 3, 70, "Input",
 CellID->1148779652],
Cell[55372, 2123, 1366, 32, 251, "Output",
 CellID->1655537558]
}, Open  ]],
Cell[56753, 2158, 189, 4, 70, "Text",
 CellID->240503747],
Cell[56945, 2164, 707, 19, 70, "Input",
 CellID->279343591],
Cell[57655, 2185, 248, 5, 70, "Text",
 CellID->1656835346],
Cell[CellGroupData[{
Cell[57928, 2194, 206, 6, 70, "Input",
 CellID->667476263],
Cell[58137, 2202, 318, 10, 39, "Output",
 CellID->1171477638]
}, Open  ]],
Cell[58470, 2215, 171, 4, 70, "Text",
 CellID->1188642071],
Cell[CellGroupData[{
Cell[58666, 2223, 121, 3, 70, "Input",
 CellID->2038036524],
Cell[58790, 2228, 1270, 30, 248, "Output",
 CellID->1847780704]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[60121, 2265, 72, 2, 70, "Section",
 CellTags->"s:3",
 CellID->1037467767],
Cell[60196, 2269, 159, 5, 70, "Text",
 CellID->1535510694],
Cell[60358, 2276, 515, 10, 70, "BulletedText",
 CellID->1318156515],
Cell[60876, 2288, 105, 2, 70, "BulletedText",
 CellID->2142122243],
Cell[CellGroupData[{
Cell[61006, 2294, 223, 5, 70, "BulletedText",
 CellID->1660839353],
Cell[61232, 2301, 619, 15, 70, "Text",
 CellID->819170424],
Cell[61854, 2318, 275, 8, 70, "Text",
 CellID->2046104655],
Cell[62132, 2328, 231, 8, 70, "Text",
 CellID->1109969164]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[62412, 2342, 71, 2, 70, "Section",
 CellTags->"s:4",
 CellID->170030189],
Cell[62486, 2346, 1457, 43, 70, "DefinitionBox3Col",
 CellID->508986919],
Cell[63946, 2391, 141, 5, 70, "Caption",
 CellID->1552834611]
}, Open  ]]
}, Open  ]],
Cell[64114, 2400, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

