(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     58726,       1694]
NotebookOptionsPosition[     50253,       1425]
NotebookOutlinePosition[     53198,       1499]
CellTagsIndexPosition[     53024,       1491]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Advanced Numerical Differential Equation Solving in \
Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/NDSolveOverview"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Advanced Numerical Differential Equation Solving in \
Mathematica\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"InterpolatingFunction" :> 
          Documentation`HelpLookup["paclet:ref/InterpolatingFunction"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"InterpolatingFunction\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Utility Packages for Numerical Differential Equation Solving", "Title",
 CellID->2011404400],

Cell[CellGroupData[{

Cell["InterpolatingFunctionAnatomy", "Section",
 CellTags->{"s:1", "InterpolatingFunctionAnatomy"},
 CellID->120436095],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 StyleBox[" ", "MR"],
 "returns solutions as ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " objects. Most of the time, simply using these as functions does what is \
needed, but occasionally it is useful to access the data inside, which \
includes the actual values and points ",
 StyleBox["NDSolve", "MR"],
 " computed when taking steps. The exact structure of an ",
 StyleBox["InterpolatingFunction", "MR"],
 " object is arranged to make the data storage efficient and evaluation at a \
given point fast. This structure may change between ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " versions, so code which is written in terms of accessing parts of \
InterpolatingFunction objects may not work with new versions of ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 ".",
 " ",
 "The ",
 StyleBox["DifferentialEquations`InterpolatingFunctionAnatomy`", "MR"],
 " package provides an interface to the data in an InterpolatingFunction \
object which will be maintained for future ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " versions."
}], "Text",
 CellID->125963850],

Cell[BoxData[GridBox[{
   {
    RowBox[{"InterpolatingFunctionDomain", "[", 
     StyleBox["ifun", "TI"], "]"}], Cell[TextData[{
     "Return a list with the domain of definition for each of the dimensions \
of the ",
     Cell[BoxData[
      ButtonBox["InterpolatingFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
     " object ",
     Cell[BoxData[
      StyleBox["ifun", "TI"]], "InlineFormula"],
     "."
    }], "TableText"]},
   {
    RowBox[{"InterpolatingFunctionCoordinates", "[", 
     StyleBox["ifun", "TI"], "]"}], Cell[TextData[{
     "Return a list with the coordinates at which data is specified in each \
of the dimensions for the ",
     Cell[BoxData[
      ButtonBox["InterpolatingFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
     " object ",
     Cell[BoxData[
      StyleBox["ifun", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {
    RowBox[{"InterpolatingFunctionGrid", "[", 
     StyleBox["ifun", "TI"], "]"}], Cell[TextData[{
     "Return the grid of points at which data is specified for ",
     "the ",
     Cell[BoxData[
      ButtonBox["InterpolatingFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
     " object ",
     Cell[BoxData[
      StyleBox["ifun", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {
    RowBox[{"InterpolatingFunctionValuesOnGrid", "[", 
     StyleBox["ifun", "TI"], "]"}], Cell[TextData[{
     "Return the values which would be returned by evaluating the ",
     Cell[BoxData[
      ButtonBox["InterpolatingFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
     " object ",
     Cell[BoxData[
      StyleBox["ifun", "TI"]], "InlineFormula"],
     " at each of its grid points."
    }], "TableText"]},
   {
    RowBox[{"InterpolatingFunctionInterpolationOrder", "[", 
     StyleBox["ifun", "TI"], "]"}], Cell[TextData[{
     "Return the interpolation order used for each of the dimensions for ",
     "the ",
     Cell[BoxData[
      ButtonBox["InterpolatingFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
     " object ",
     Cell[BoxData[
      StyleBox["ifun", "TI"]], "InlineFormula"],
     "."
    }], "TableText"]},
   {
    RowBox[{"InterpolatingFunctionDerivativeOrder", "[", 
     StyleBox["ifun", "TI"], "]"}], Cell[TextData[{
     "Return the order of the derivative of the base function for which \
values are specified when evaluating ",
     "the ",
     Cell[BoxData[
      ButtonBox["InterpolatingFunction",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
     " object ",
     Cell[BoxData[
      StyleBox["ifun", "TI"]], "InlineFormula"],
     "."
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{GridBoxItemSize->{"Columns" -> {{
      Scaled[0.5]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}},
 CellID->1577784482],

Cell[TextData[{
 "Anatomy of ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " objects."
}], "Caption",
 CellID->1599341692],

Cell["This loads the package.", "MathCaption",
 CellID->1232983610],

Cell[BoxData[
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`InterpolatingFunctionAnatomy`\>\"",
    "]"}], ";"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->408132437],

Cell[TextData[{
 "One common situation where the ",
 StyleBox["InterpolatingFunctionAnatomy", "MR"],
 " package is useful is when ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " cannot compute a solution over the full range of values that you \
specified, and you want to plot all of the solution which was computed to try \
to understand better when might have gone wrong."
}], "Text",
 CellID->2046748285],

Cell["\<\
Here is an example of a differential equation which cannot be computed up to \
the specified endpoint. \
\>", "MathCaption",
 CellID->1722083948],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ifun", " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{"x", " ", "/.", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{
          RowBox[{"x", "'"}], "[", "t", "]"}], " ", "\[Equal]", " ", 
         RowBox[{
          RowBox[{"Exp", "[", 
           RowBox[{"x", "[", "t", "]"}], "]"}], " ", "-", " ", 
          RowBox[{"x", "[", "t", "]"}]}]}], ",", " ", 
        RowBox[{
         RowBox[{"x", "[", "0", "]"}], " ", "\[Equal]", " ", "1"}]}], "}"}], 
      ",", "x", ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "10"}], "}"}]}], "]"}]}], "]"}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->547648506],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ndsz\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"At \\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\
\\(t\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" == \\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(0.5160191740198964`\\)\\!\\(\\*StyleBox[\\\"\\\\\\\", \
step size is effectively zero; singularity or stiff system suspected.\\\\\\\"\
\\\", \\\"MSG\\\"]\\) \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet://Messages/NDSolve/ndsz\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellLabel->"During evaluation of In[2]:=",
 CellID->1457649098],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0.`", ",", "0.5160191740198964`"}], "}"}], "}"}], 
    ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{300, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->933234494]
}, Open  ]],

Cell["This gets the domain.", "MathCaption",
 CellID->790126763],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"domain", " ", "=", " ", 
  RowBox[{"InterpolatingFunctionDomain", "[", "ifun", "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1337859063],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"0.`", ",", "0.5160191740198964`"}], "}"}], "}"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1764776009]
}, Open  ]],

Cell[TextData[{
 "Once the domain has been returned in a list, it is easy to use ",
 Cell[BoxData[
  ButtonBox["Part",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Part"]], "InlineFormula"],
 " to get the desired endpoints and make the plot."
}], "MathCaption",
 CellID->279096065],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"begin", ",", " ", "end"}], "}"}], " ", "=", " ", 
   RowBox[{"domain", "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"ifun", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "begin", ",", " ", "end"}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->668648379],

Cell[BoxData[
 GraphicsBox[{{}, {}, {
    {Hue[0.67, 0.6, 0.6]}, LineBox[CompressedData["
1:eJwBQQS++yFib1JlAgAAAEMAAAACAAAAwD5c/XqdRj6zftsEAADwP34mCfK8
viQ/Heu4Ox0B8D/GKxO3j740P8/XkIM6AvA/ai6YGXm+RD8aWdRGdQTwP7yv
2sptvlQ/50zWvOsI8D9l8HsjaL5kP6igQ2fcEfA/uZDMT2W+dD8OF6vSzCPw
P+Pg9OVjvoQ/w2XlwupH8D8UObVWup2VP0D6xRRal/A/oQNNqcAOoD/2kfAo
+OLwP0Ef6opHNKU/bvDI2IYu8T/uUl3/ecmqP3eBklApgvE/Yonlv0v/rz8f
IkCS3dHxP/LroYlk0rI/EHBugiMq8j93bSbl9Je1PxcVbkjTgvI/X3C15tQt
uD8ICvGHddfyP05/r7GK+7o/6snqTm418z+gD7QikJm9P8vN0CxXj/M/G/3G
ojMVwD9JF1UppenzP2p46RiKecE/i0nkPDFO9D9qNBFiCMbCP/uGp9WQrvQ/
bXbuj3EuxD9189i0/hn1P5NltpZDkMU/Pfduw6eG9T9rlYNwPdrGPx8LQ64J
7/U/RksGLyJAyD9rbUi8zWP2P9JBjsAujsk/ub/EclvU9j9hvss2JvjKP30o
BUt+Uvc/E+jzhYZbzD8Lz1zx4dL3P3dSIagOp80/DXYIpQpP+D/eQgSvgQ7P
P20xda7T2vg/+zl2RA4v0D/xnUUTmWL5Pxip3x2Q09A/5mJH+Q3t+T8326Np
B4bRP2NlYfjIifo/rq3qnpIs0j8Fc2KkvCL7PydDjEYT4dI/SyF0dYfQ+z8x
L6NaSJLTP9JhZ0QchPw/lLs8WJE31D9T38zEezT9P/kKMcjP6tQ/LddG+on+
/T+2+qchIpLVP+0Sbyh7xv4/BEGU5yg21j+J5/Q0bJb/P1RK2x8l6NY/37xz
EJlDAED986RBNY7XPxjb0xMFvABAqGDJ1TpC2D+TcpbExEgBQKttcFNU6tg/
47nOq1rXAUBA0Yw9Io/ZPxTVpxN1bwJA1vcDmuVB2j9TOmXI5CQDQMW+/d+8
6No/0H4qa/PgA0C2SFKYiZ3bP/XeLVEmxgRAOCkcvQpP3D+ThzTAA8gFQBOq
aMuf9Nw/MyJEkTffBkABTLwLZU7dP4CDBlO2igdA7+0PTCqo3T8oQwib/UcI
QArgJIH5+90/nZPUCz0MCUAXWa8b4SXeP8ODHJ7HdglAJNI5tshP3j80R135
0OcJQIdviSHyod4/VY77nZTbCkA4PjHXBsvePwXEgMGAYgtA6gzZjBv03j8M
TimQ2/MLQFyMF5+tIN8/Sn9GIVifDEDOC1axP03fP5A/nbPkWw1AQIuUw9F5
3z9gq2UfLC0OQLIK09Vjpt8/XTV4yioYD0B2niJcL7vfP7NYMhF6kA9AOjJy
4vrP3z/TzCtVZQgQQP7FwWjG5N8/+YknLB1NEEDCWRHvkfnfPwl/X4gTlxBA
9+SKZQAH4D/fRYr8vOUQQC5TDNo=
     "]]}},
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  Method->{},
  PlotRange->{{0., 0.516019174019896}, {1.00000001809523, 4.2243537387271}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{364, 228},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->1801779332]
}, Open  ]],

Cell["\<\
From the plot, it is quite apparent that a singularity has formed and it will \
not be possible to integrate the system any further.\
\>", "Text",
 CellID->1880787975],

Cell[TextData[{
 "Sometimes it is useful to see where ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " took steps.",
 " ",
 "Getting the coordinates is useful for doing this."
}], "Text",
 CellID->1705695425],

Cell[TextData[{
 "This shows the values which ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " computed at each step it took.",
 " ",
 "It is quite apparent from this that nearly all of the steps were used to \
try to resolve the singularity."
}], "MathCaption",
 CellID->1603640714],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"coords", " ", "=", " ", 
   RowBox[{"First", "[", 
    RowBox[{"InterpolatingFunctionCoordinates", "[", "ifun", "]"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"coords", ",", " ", 
     RowBox[{"ifun", "[", "coords", "]"}]}], "}"}], "]"}], "]"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->796266914],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 228},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->301071588]
}, Open  ]],

Cell["\<\
The package is particularly useful for analyzing the computed solutions of \
PDEs\
\>", "Text",
 CellID->1358176989],

Cell["\<\
With this initial condition, Burgers' equation forms a steep front.\
\>", "MathCaption",
 CellID->924120605],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"mdfun", " ", "=", " ", 
  RowBox[{"First", "[", 
   RowBox[{"u", " ", "/.", 
    RowBox[{"NDSolve", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"x", ",", "t"}], "]"}], ",", "t"}], "]"}], " ", 
         "\[Equal]", " ", 
         RowBox[{
          RowBox[{"0.01", " ", 
           RowBox[{"D", "[", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"x", ",", "t"}], "]"}], ",", "x", ",", "x"}], "]"}]}], 
          " ", "-", " ", 
          RowBox[{
           RowBox[{"u", "[", 
            RowBox[{"x", ",", "t"}], "]"}], " ", 
           RowBox[{"D", "[", 
            RowBox[{
             RowBox[{"u", "[", 
              RowBox[{"x", ",", "t"}], "]"}], ",", "x"}], "]"}]}]}]}], ",", 
        " ", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"0", ",", " ", "t"}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{"u", "[", 
          RowBox[{"1", ",", "t"}], "]"}]}], ",", " ", 
        RowBox[{
         RowBox[{"u", "[", 
          RowBox[{"x", ",", "0"}], "]"}], " ", "\[Equal]", " ", 
         RowBox[{"Sin", "[", 
          RowBox[{"2", " ", "Pi", " ", "x"}], "]"}]}]}], "}"}], ",", "u", ",",
       
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "0", ",", "0.5"}], "}"}]}], "]"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[8]:=",
 CellID->1662037880],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"ndsz\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"At \\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\
\\(t\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" == \\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(0.472151168326526`\\)\\!\\(\\*StyleBox[\\\"\\\\\\\", \
step size is effectively zero; singularity or stiff system suspected.\\\\\\\"\
\\\", \\\"MSG\\\"]\\) \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet://Messages/NDSolve/ndsz\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellLabel->"During evaluation of In[8]:=",
 CellID->1001368044],

Cell[BoxData[
 RowBox[{
  RowBox[{"NDSolve", "::", "\<\"eerr\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Warning: Scaled local spatial error \
estimate of \\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(9.135898727911074`*^12\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" \
at \\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\(t\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" = \\\
\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(0.472151168326526`\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" in \
the direction of independent variable \\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\(x\\)\
\\!\\(\\*StyleBox[\\\"\\\\\\\" is much greater than prescribed error \
tolerance. Grid spacing with \\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(27\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" points may be too \
large to achieve the desired accuracy or precision.  A singularity may have \
formed or you may want to specify a smaller grid spacing using the \
MaxStepSize or MinPoints method options.\\\\\\\"\\\", \\\"MSG\\\"]\\) \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, \
ButtonData:>\\\"paclet://Messages/NDSolve/eerr\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellLabel->"During evaluation of In[8]:=",
 CellID->1096645112],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\<\"...\"\>", ",", "0.`", ",", "1.`", ",", "\<\"...\"\>"}], 
       "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "0.472151168326526`"}], "}"}]}], "}"}], 
    ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{429, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->1475497628]
}, Open  ]],

Cell["This shows the number of points used in each dimension.", "MathCaption",
 CellID->1078690929],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Map", "[", 
  RowBox[{"Length", ",", " ", 
   RowBox[{"InterpolatingFunctionCoordinates", "[", "mdfun", "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->56907487],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"27", ",", "312"}], "}"}]], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->1913679022]
}, Open  ]],

Cell["This shows the interpolation order used in each dimension.", \
"MathCaption",
 CellID->1199127024],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingFunctionInterpolationOrder", "[", "mdfun", 
  "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->182871337],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"5", ",", "3"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->1888688746]
}, Open  ]],

Cell["\<\
This shows that the inability to resolve the front has manifested itself as \
numerical instability.\
\>", "MathCaption",
 CellID->650985069],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Max", "[", 
  RowBox[{"Abs", "[", 
   RowBox[{"InterpolatingFunctionValuesOnGrid", "[", "mdfun", "]"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[11]:=",
 CellID->1415854947],

Cell[BoxData["1.1492807485974238`*^12"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->353050808]
}, Open  ]],

Cell["\<\
This shows the values computed at the spatial grid points at the endpoint of \
the temporal integration.\
\>", "MathCaption",
 CellID->550249706],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"end", " ", "=", " ", 
   RowBox[{
    RowBox[{"InterpolatingFunctionDomain", "[", "mdfun", "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", 
      RowBox[{"-", "1"}]}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"X", " ", "=", " ", 
   RowBox[{
    RowBox[{"InterpolatingFunctionCoordinates", "[", "mdfun", "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"X", ",", " ", 
      RowBox[{"mdfun", "[", 
       RowBox[{"X", ",", " ", "end"}], "]"}]}], "}"}], "]"}], ",", " ", 
   RowBox[{"PlotStyle", "\[Rule]", 
    RowBox[{"PointSize", "[", ".025", "]"}]}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "1"}], "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[12]:=",
 CellID->990455248],

Cell[BoxData[
 GraphicsBox[{
   {Hue[0.67, 0.6, 0.6], PointSize[0.025], 
    PointBox[{0.653846153846154, -0.885619897364659}], 
    PointBox[{0.615384615384615, -0.0726705755619866}], 
    PointBox[{0.730769230769231, -0.425760558518071}], 
    PointBox[{0.692307692307692, -0.4276453966533}], 
    PointBox[{0.538461538461538, -0.338806390762329}], 
    PointBox[{0.5, -1.14928074859742*^12}], 
    PointBox[{0.576923076923077, -6.56694115319959*^6}], 
    PointBox[{0.923076923076923, -0.120219600176053}], 
    PointBox[{0.884615384615385, -0.180137214019438}], 
    PointBox[{1., -2.28822526405246*^-7}], 
    PointBox[{0.961538461538462, -0.0601457764992652}], 
    PointBox[{0.807692307692308, -0.299051289122987}], 
    PointBox[{0.769230769230769, -0.357714465437724}], 
    PointBox[{0.846153846153846, -0.240006050995014}], 
    PointBox[{0.153846153846154, 0.239846590124078}], 
    PointBox[{0.115384615384615, 0.180145784062172}], 
    PointBox[{0.230769230769231, 0.358207850956085}], 
    PointBox[{0.192307692307692, 0.299233875220426}], 
    PointBox[{0.0384615384615385, 0.0601454297332885}], 
    PointBox[{0., -2.28822526402446*^-7}], 
    PointBox[{0.0769230769230769, 0.120218932242074}], 
    PointBox[{0.423076923076923, 0.57750591665398}], 
    PointBox[{0.384615384615385, 0.643204756121204}], 
    PointBox[{0.461538461538462, 1.4364159171178*^11}], 
    PointBox[{0.307692307692308, 0.474399374645502}], 
    PointBox[{0.269230769230769, 0.416649300878975}], 
    PointBox[{0.346153846153846, 0.531114683526867}]}, {}},
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotRange->{{0., 1.}, {-1, 1}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], Automatic}]], "Output",
 ImageSize->{364, 225},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->415066987]
}, Open  ]],

Cell["\<\
It is easily seen from the point plot above that the front has not been \
resolved.\
\>", "Text",
 CellID->230253267],

Cell["\<\
This makes a 3-D plot showing how the time evolution for each of the spatial \
grid points. The initial condition is shown in red.\
\>", "MathCaption",
 CellID->180877962],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Show", "[", 
  RowBox[{
   RowBox[{"Graphics3D", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Map", "[", 
       RowBox[{"Line", ",", " ", 
        RowBox[{"MapThread", "[", 
         RowBox[{"Append", ",", " ", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"InterpolatingFunctionGrid", "[", "mdfun", "]"}], ",", 
            " ", 
            RowBox[{
            "InterpolatingFunctionValuesOnGrid", "[", "mdfun", "]"}]}], "}"}],
           ",", " ", "2"}], "]"}]}], "]"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"RGBColor", "[", 
         RowBox[{"1", ",", "0", ",", "0"}], "]"}], ",", 
        RowBox[{"Line", "[", 
         RowBox[{"Transpose", "[", 
          RowBox[{"{", 
           RowBox[{"X", ",", " ", 
            RowBox[{"0.", " ", "X"}], ",", " ", 
            RowBox[{"mdfun", "[", 
             RowBox[{"X", ",", "0."}], "]"}]}], "}"}], "]"}], "]"}]}], 
       "}"}]}], "}"}], "]"}], ",", " ", 
   RowBox[{"BoxRatios", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], ",", " ", 
   RowBox[{"PlotRange", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"All", ",", " ", "All", ",", " ", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->1626691781],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 391},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->160241047]
}, Open  ]],

Cell[TextData[{
 "When a derivative is taken of an ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object, a new ",
 StyleBox["InterpolatingFunction", "MR"],
 " object is returned which gives the requested derivative when evaluated at \
a point.",
 " ",
 "The ",
 StyleBox["InterpolatingFunctionDerivativeOrder", "MR"],
 " is a way of determining what derivative will be evaluated."
}], "Text",
 CellID->432962113],

Cell[TextData[{
 "The derivative returns a new ",
 Cell[BoxData[
  ButtonBox["InterpolatingFunction",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InterpolatingFunction"]], "InlineFormula"],
 " object."
}], "MathCaption",
 CellID->1905787846],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"dmdfun", " ", "=", " ", 
  RowBox[{
   RowBox[{"Derivative", "[", 
    RowBox[{"0", ",", "1"}], "]"}], "[", "mdfun", "]"}]}]], "Input",
 CellLabel->"In[16]:=",
 CellID->89277061],

Cell[BoxData[
 TagBox[
  RowBox[{"InterpolatingFunction", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\<\"...\"\>", ",", "0.`", ",", "1.`", ",", "\<\"...\"\>"}], 
       "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "0.472151168326526`"}], "}"}]}], "}"}], 
    ",", "\<\"<>\"\>"}], "]"}],
  False,
  Editable->False]], "Output",
 ImageSize->{429, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->1798233202]
}, Open  ]],

Cell["This shows what derivative will be evaluated.", "MathCaption",
 CellID->1639092173],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"InterpolatingFunctionDerivativeOrder", "[", "dmdfun", 
  "]"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->1794972486],

Cell[BoxData[
 RowBox[{"Derivative", "[", 
  RowBox[{"0", ",", "1"}], "]"}]], "Output",
 ImageSize->{114, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->220645034]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["NDSolveUtilities", "Section",
 CellTags->{"s:2", "NDSolveUtilities"},
 CellID->287875440],

Cell[TextData[{
 "A number of utility routines have been written to facilitate the \
investigation and comparison of various ",
 StyleBox["NDSolve", "MR"],
 " methods. These functions have been collected in the package ",
 StyleBox["DifferentialEquations`NDSolveUtilities`", "MR"],
 "."
}], "Text",
 CellID->2134588770],

Cell[BoxData[GridBox[{
   {
    RowBox[{"CompareMethods", "[", 
     RowBox[{
      StyleBox["sys", "TI"], ",", 
      StyleBox["refsol", "TI"], ",", 
      StyleBox["methods", "TI"], ",", 
      StyleBox["opts", "TI"]}], "]"}], Cell[TextData[{
     "Return statistics for various methods applied to the system ",
     StyleBox["sys", "TI"],
     "."
    }], "TableText"]},
   {
    RowBox[{"FinalSolutions", "[", 
     RowBox[{
      StyleBox["sys", "TI"], ",", 
      StyleBox["sols", "TI"]}], "]"}], Cell[TextData[{
     "Return the solution values at the end of the numerical integration for \
various solutions ",
     StyleBox["sols", "TI"],
     "  corresponding to the system ",
     StyleBox["sys", "TI"],
     "."
    }], "TableText"]},
   {
    RowBox[{"InvariantErrorPlot", "[", 
     RowBox[{
      StyleBox["invts", "TI"], ",", 
      StyleBox["dvars", "TI"], ",", 
      StyleBox["ivar", "TI"], ",", 
      StyleBox["sol", "TI"], ",", 
      StyleBox["opts", "TI"]}], "]"}], Cell[TextData[{
     "Return a plot of the error in the invariants ",
     StyleBox["invts", "TI"],
     " for the solution ",
     StyleBox["sol", "TI"],
     "."
    }], "TableText"]},
   {
    RowBox[{"RungeKuttaLinearStabilityFunction", "[", 
     RowBox[{
      StyleBox["amat", "TI"], ",", 
      StyleBox["bvec", "TI"], ",", 
      StyleBox["var", "TI"]}], "]"}], Cell[TextData[{
     "Return the linear stability function for the Runge\[LongDash]Kutta \
method with coefficient matrix ",
     StyleBox["amat", "TI"],
     ", weight vector ",
     StyleBox["bvec", "TI"],
     " using the variable ",
     StyleBox["var", "TI"],
     "."
    }], "TableText"]},
   {
    RowBox[{"StepDataPlot", "[", 
     RowBox[{
      StyleBox["sols", "TI"], ",", 
      StyleBox["opts", "TI"]}], "]"}], Cell[TextData[{
     "Return plots of the step sizes taken for the solutions ",
     StyleBox["sols", "TI"],
     "."
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{GridBoxItemSize->{"Columns" -> {{
      Scaled[0.5]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}},
 CellID->1435310699],

Cell[TextData[{
 "Functions provided in the ",
 StyleBox["NDSolveUtilities", "MR"],
 " package."
}], "Caption",
 CellID->1211996045],

Cell["This loads the package.", "MathCaption",
 CellID->1649143002],

Cell[BoxData[
 RowBox[{"Needs", "[", "\"\<DifferentialEquations`NDSolveUtilities`\>\"", 
  "]"}]], "Input",
 CellLabel->"In[18]:=",
 CellID->972453996],

Cell[TextData[{
 "A useful means of analysing Runge\[LongDash]Kutta methods is to study how \
they behave when applied to a scalar linear test problem (see the package ",
 StyleBox["FunctionApproximations.m", "MR"],
 ")."
}], "Text",
 CellID->515637456],

Cell["\<\
This assigns the (exact or infinitely precise) coefficients for the 2 stage \
implicit Runge\[LongDash]Kutta Gauss method of order 4.\
\>", "MathCaption",
 CellID->88221842],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"amat", ",", " ", "bvec", ",", " ", "cvec"}], "}"}], " ", "=", 
  RowBox[{"NDSolve`ImplicitRungeKuttaGaussCoefficients", "[", 
   RowBox[{"4", ",", " ", "Infinity"}], "]"}]}]], "Input",
 CellLabel->"In[19]:=",
 CellID->2051144926],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       FractionBox["1", "4"], ",", 
       RowBox[{
        FractionBox["1", "12"], " ", 
        RowBox[{"(", 
         RowBox[{"3", "-", 
          RowBox[{"2", " ", 
           SqrtBox["3"]}]}], ")"}]}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        FractionBox["1", "12"], " ", 
        RowBox[{"(", 
         RowBox[{"3", "+", 
          RowBox[{"2", " ", 
           SqrtBox["3"]}]}], ")"}]}], ",", 
       FractionBox["1", "4"]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     FractionBox["1", "2"], ",", 
     FractionBox["1", "2"]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      FractionBox["1", "6"], " ", 
      RowBox[{"(", 
       RowBox[{"3", "-", 
        SqrtBox["3"]}], ")"}]}], ",", 
     RowBox[{
      FractionBox["1", "6"], " ", 
      RowBox[{"(", 
       RowBox[{"3", "+", 
        SqrtBox["3"]}], ")"}]}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{532, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->1220526357]
}, Open  ]],

Cell["\<\
This computes the linear stability function, which corresponds to the (2,2) \
Pad\[EAcute] approximation to the exponential at the origin.\
\>", "MathCaption",
 CellID->538705082],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RungeKuttaLinearStabilityFunction", "[", 
  RowBox[{"amat", ",", " ", "bvec", ",", " ", "z"}], "]"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->1102788302],

Cell[BoxData[
 FractionBox[
  RowBox[{"1", "+", 
   FractionBox["z", "2"], "+", 
   FractionBox[
    SuperscriptBox["z", "2"], "12"]}], 
  RowBox[{"1", "-", 
   FractionBox["z", "2"], "+", 
   FractionBox[
    SuperscriptBox["z", "2"], "12"]}]]], "Output",
 ImageSize->{66, 53},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->1869716248]
}, Open  ]],

Cell[TextData[{
 "Examples of the functions ",
 StyleBox["CompareMethods", "MR"],
 ", ",
 StyleBox["FinalSolutions", "MR"],
 ", ",
 StyleBox["RungeKuttaLinearStabilityFunction", "MR"],
 " and ",
 StyleBox["StepDataPlot", "MR"],
 " can be found within ",
 "\"",
 ButtonBox["ExplicitRungeKutta Method for NDSolve",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveExplicitRungeKutta"],
 "\"."
}], "Text",
 CellID->419489156],

Cell[TextData[{
 "Examples of the function ",
 StyleBox["InvariantErrorPlot", "MR"],
 " can be found within \"",
 ButtonBox["Projection Method for NDSolve",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveProjection"],
 "\"."
}], "Text",
 CellID->1753773371],

Cell[CellGroupData[{

Cell["InvariantErrorPlot Options", "Subsection",
 CellTags->"s:1",
 CellID->23787580],

Cell[TextData[{
 "The function ",
 StyleBox["InvariantErrorPlot", "MR"],
 " has a number of options that can be used to control the form of the \
result."
}], "Text",
 CellID->708405668],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"InvariantDimensions", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies the dimensions of the invariants", "TableText"]},
   {"InvariantErrorFunction", 
    RowBox[{
     RowBox[{
      ButtonBox["Abs",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Abs"], "[", 
      RowBox[{
       ButtonBox["Subtract",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Subtract"], "[", 
       RowBox[{"#1", ",", "#2"}], "]"}], "]"}], "&"}], Cell[
    "specifies the function to use for comparing errors", "TableText"]},
   {"InvariantErrorSampleRate", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "specifies how often errors are sampled", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->156539429],

Cell[TextData[{
 "Options of the function ",
 Cell[BoxData["InvariantErrorPlot"], "InlineFormula"],
 ". "
}], "Caption",
 CellID->1014242828],

Cell[TextData[{
 "The default value for ",
 StyleBox["InvariantDimensions", "MR"],
 " is to determine the dimensions from the structure of the input, ",
 StyleBox["Dimensions[", "MR"],
 StyleBox["invts", "TI"],
 StyleBox["]", "MR"],
 "."
}], "Text",
 CellID->1544970685],

Cell[TextData[{
 "The default value for ",
 StyleBox["InvariantErrorFunction", "MR"],
 " is a function to compute the absolute error."
}], "Text",
 CellID->386792696],

Cell[TextData[{
 "The default value for ",
 StyleBox["InvariantErrorSampleRate", "MR"],
 " is to sample all points if there are less than 1000 steps taken. Above \
this threshold a logarithmic sample rate is used."
}], "Text",
 CellID->1195120790],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->156160542],

Cell[TextData[{
 ButtonBox["Advanced Numerical Differential Equation Solving in ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
 StyleBox[ButtonBox["Mathematica",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveOverview"],
  FontSlant->"Italic"]
}], "RelatedTutorials",
 CellID->604721514]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Utility Packages for Numerical Differential Equation Solving - \
Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 39, 57.2590478}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "NDSolve returns solutions as InterpolatingFunction objects. Most of the \
time, simply using these as functions does what is needed, but occasionally \
it is useful to access the data inside, which includes the actual values and \
points NDSolve computed when taking steps. The exact structure of an \
InterpolatingFunction object is arranged to make the data storage efficient \
and evaluation at a given point fast. This structure may change between \
Mathematica versions, so code which is written in terms of accessing parts of \
InterpolatingFunction objects may not work with new versions of Mathematica. \
The DifferentialEquations`InterpolatingFunctionAnatomy` package provides an \
interface to the data in an InterpolatingFunction object which will be \
maintained for future Mathematica versions. Anatomy of InterpolatingFunction \
objects. This loads the package.", "synonyms" -> {}, "title" -> 
    "Utility Packages for Numerical Differential Equation Solving", "type" -> 
    "Tutorial", "uri" -> "tutorial/NDSolvePackages"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "s:1"->{
  Cell[2046, 65, 119, 2, 70, "Section",
   CellTags->{"s:1", "InterpolatingFunctionAnatomy"},
   CellID->120436095],
  Cell[47510, 1326, 85, 2, 70, "Subsection",
   CellTags->"s:1",
   CellID->23787580]},
 "InterpolatingFunctionAnatomy"->{
  Cell[2046, 65, 119, 2, 70, "Section",
   CellTags->{"s:1", "InterpolatingFunctionAnatomy"},
   CellID->120436095]},
 "s:2"->{
  Cell[41173, 1095, 95, 2, 70, "Section",
   CellTags->{"s:2", "NDSolveUtilities"},
   CellID->287875440]},
 "NDSolveUtilities"->{
  Cell[41173, 1095, 95, 2, 70, "Section",
   CellTags->{"s:2", "NDSolveUtilities"},
   CellID->287875440]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"s:1", 52394, 1469},
 {"InterpolatingFunctionAnatomy", 52634, 1476},
 {"s:2", 52762, 1480},
 {"NDSolveUtilities", 52893, 1484}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1298, 33, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[1923, 60, 98, 1, 70, "Title",
 CellID->2011404400],
Cell[CellGroupData[{
Cell[2046, 65, 119, 2, 70, "Section",
 CellTags->{"s:1", "InterpolatingFunctionAnatomy"},
 CellID->120436095],
Cell[2168, 69, 1325, 35, 70, "Text",
 CellID->125963850],
Cell[3496, 106, 3109, 87, 70, "DefinitionBox",
 CellID->1577784482],
Cell[6608, 195, 225, 8, 70, "Caption",
 CellID->1599341692],
Cell[6836, 205, 67, 1, 70, "MathCaption",
 CellID->1232983610],
Cell[6906, 208, 185, 6, 70, "Input",
 CellID->408132437],
Cell[7094, 216, 482, 12, 70, "Text",
 CellID->2046748285],
Cell[7579, 230, 155, 4, 70, "MathCaption",
 CellID->1722083948],
Cell[CellGroupData[{
Cell[7759, 238, 718, 22, 70, "Input",
 CellID->547648506],
Cell[8480, 262, 632, 12, 70, "Message",
 CellID->1457649098],
Cell[9115, 276, 381, 14, 36, "Output",
 CellID->933234494]
}, Open  ]],
Cell[9511, 293, 64, 1, 70, "MathCaption",
 CellID->790126763],
Cell[CellGroupData[{
Cell[9600, 298, 165, 4, 70, "Input",
 CellID->1337859063],
Cell[9768, 304, 249, 8, 36, "Output",
 CellID->1764776009]
}, Open  ]],
Cell[10032, 315, 285, 8, 70, "MathCaption",
 CellID->279096065],
Cell[CellGroupData[{
Cell[10342, 327, 407, 13, 70, "Input",
 CellID->668648379],
Cell[10752, 342, 2010, 43, 249, "Output",
 CellID->1801779332]
}, Open  ]],
Cell[12777, 388, 177, 4, 70, "Text",
 CellID->1880787975],
Cell[12957, 394, 282, 10, 70, "Text",
 CellID->1705695425],
Cell[13242, 406, 357, 11, 70, "MathCaption",
 CellID->1603640714],
Cell[CellGroupData[{
Cell[13624, 421, 406, 12, 47, "Input",
 CellID->796266914],
Cell[14033, 435, 2522, 45, 249, "Output",
 Evaluatable->False,
 CellID->301071588]
}, Open  ]],
Cell[16570, 483, 126, 4, 70, "Text",
 CellID->1358176989],
Cell[16699, 489, 118, 3, 70, "MathCaption",
 CellID->924120605],
Cell[CellGroupData[{
Cell[16842, 496, 1528, 46, 70, "Input",
 CellID->1662037880],
Cell[18373, 544, 631, 12, 70, "Message",
 CellID->1001368044],
Cell[19007, 558, 1167, 21, 70, "Message",
 CellID->1096645112],
Cell[20177, 581, 516, 18, 36, "Output",
 CellID->1475497628]
}, Open  ]],
Cell[20708, 602, 99, 1, 70, "MathCaption",
 CellID->1078690929],
Cell[CellGroupData[{
Cell[20832, 607, 198, 6, 70, "Input",
 CellID->56907487],
Cell[21033, 615, 207, 7, 36, "Output",
 CellID->1913679022]
}, Open  ]],
Cell[21255, 625, 104, 2, 70, "MathCaption",
 CellID->1199127024],
Cell[CellGroupData[{
Cell[21384, 631, 143, 4, 70, "Input",
 CellID->182871337],
Cell[21530, 637, 205, 7, 36, "Output",
 CellID->1888688746]
}, Open  ]],
Cell[21750, 647, 151, 4, 70, "MathCaption",
 CellID->650985069],
Cell[CellGroupData[{
Cell[21926, 655, 199, 6, 70, "Input",
 CellID->1415854947],
Cell[22128, 663, 181, 5, 36, "Output",
 CellID->353050808]
}, Open  ]],
Cell[22324, 671, 155, 4, 70, "MathCaption",
 CellID->550249706],
Cell[CellGroupData[{
Cell[22504, 679, 909, 27, 70, "Input",
 CellID->990455248],
Cell[23416, 708, 1897, 42, 246, "Output",
 CellID->415066987]
}, Open  ]],
Cell[25328, 753, 127, 4, 70, "Text",
 CellID->230253267],
Cell[25458, 759, 181, 4, 70, "MathCaption",
 CellID->180877962],
Cell[CellGroupData[{
Cell[25664, 767, 1360, 39, 83, "Input",
 CellID->1626691781],
Cell[27027, 808, 12076, 202, 412, "Output",
 Evaluatable->False,
 CellID->160241047]
}, Open  ]],
Cell[39118, 1013, 517, 15, 70, "Text",
 CellID->432962113],
Cell[39638, 1030, 246, 8, 70, "MathCaption",
 CellID->1905787846],
Cell[CellGroupData[{
Cell[39909, 1042, 202, 6, 70, "Input",
 CellID->89277061],
Cell[40114, 1050, 517, 18, 36, "Output",
 CellID->1798233202]
}, Open  ]],
Cell[40646, 1071, 89, 1, 70, "MathCaption",
 CellID->1639092173],
Cell[CellGroupData[{
Cell[40760, 1076, 142, 4, 70, "Input",
 CellID->1794972486],
Cell[40905, 1082, 219, 7, 36, "Output",
 CellID->220645034]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[41173, 1095, 95, 2, 70, "Section",
 CellTags->{"s:2", "NDSolveUtilities"},
 CellID->287875440],
Cell[41271, 1099, 319, 8, 70, "Text",
 CellID->2134588770],
Cell[41593, 1109, 2112, 66, 70, "DefinitionBox",
 CellID->1435310699],
Cell[43708, 1177, 132, 5, 70, "Caption",
 CellID->1211996045],
Cell[43843, 1184, 67, 1, 70, "MathCaption",
 CellID->1649143002],
Cell[43913, 1187, 151, 4, 70, "Input",
 CellID->972453996],
Cell[44067, 1193, 253, 6, 70, "Text",
 CellID->515637456],
Cell[44323, 1201, 183, 4, 70, "MathCaption",
 CellID->88221842],
Cell[CellGroupData[{
Cell[44531, 1209, 281, 7, 70, "Input",
 CellID->2051144926],
Cell[44815, 1218, 1151, 43, 51, "Output",
 CellID->1220526357]
}, Open  ]],
Cell[45981, 1264, 189, 4, 70, "MathCaption",
 CellID->538705082],
Cell[CellGroupData[{
Cell[46195, 1272, 180, 4, 70, "Input",
 CellID->1102788302],
Cell[46378, 1278, 388, 14, 74, "Output",
 CellID->1869716248]
}, Open  ]],
Cell[46781, 1295, 432, 16, 70, "Text",
 CellID->419489156],
Cell[47216, 1313, 269, 9, 70, "Text",
 CellID->1753773371],
Cell[CellGroupData[{
Cell[47510, 1326, 85, 2, 70, "Subsection",
 CellTags->"s:1",
 CellID->23787580],
Cell[47598, 1330, 186, 6, 70, "Text",
 CellID->708405668],
Cell[47787, 1338, 1126, 30, 70, "DefinitionBox3Col",
 CellID->156539429],
Cell[48916, 1370, 141, 5, 70, "Caption",
 CellID->1014242828],
Cell[49060, 1377, 270, 9, 70, "Text",
 CellID->1544970685],
Cell[49333, 1388, 166, 5, 70, "Text",
 CellID->386792696],
Cell[49502, 1395, 247, 6, 70, "Text",
 CellID->1195120790],
Cell[CellGroupData[{
Cell[49774, 1405, 72, 1, 70, "RelatedTutorialsSection",
 CellID->156160542],
Cell[49849, 1408, 326, 9, 70, "RelatedTutorials",
 CellID->604721514]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[50226, 1423, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

