(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    137138,       3068]
NotebookOptionsPosition[    126570,       2725]
NotebookOutlinePosition[    130772,       2833]
CellTagsIndexPosition[    130483,       2821]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], " "}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Projection Method for NDSolve", "Title",
 CellTags->"c:1",
 CellID->947483451],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellTags->"s:1",
 CellID->1608456566],

Cell["\<\
When a differential system has a certain structure, it is advantageous if a \
numerical integration method preserves the structure. In certain situations \
it is useful to solve differential equations in which solutions are \
constrained. Projection methods work by taking a time step with a numerical \
integration method and then projecting the approximate solution onto the \
manifold on which the true solution evolves.\
\>", "Text",
 CellID->1904480722],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " includes a differential algebraic solver which may be appropriate and is \
described in more detail within ",
 ButtonBox["Numerical Solution of Differential-Algebraic Equations",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveIntroductoryTutorialDAEs"],
 "."
}], "Text",
 CellID->662484585],

Cell[TextData[{
 "Sometimes the form of the equations may not be reduced to the form required \
by a DAE solver. Furthermore so-called index reduction techniques can destroy \
certain structural properties, such as symplecticity, that the differential \
system may possess (see [",
 ButtonBox["HW96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#32197"],
 "] and [",
 ButtonBox["HLW02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2433"],
 "]). An example that illustrates this can be found in the documentation for \
DAEs."
}], "Text",
 CellID->1676766389],

Cell[TextData[{
 "In such cases it is often possible to solve a differential system and then \
use a projective procedure to ensure that the constraints are conserved. This \
is the idea behind the method ",
 StyleBox["Projection", "MR"],
 "."
}], "Text",
 CellID->1689566873],

Cell[TextData[{
 "If the differential system is \[Rho]-reversible then a ",
 StyleBox["symmetric",
  FontSlant->"Italic"],
 " projection process can be advantageous (see [",
 ButtonBox["H00",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#25443"],
 "]). Symmetric projection is generally more costly than projection and has \
not yet been implemented in ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->17520324],

Cell[CellGroupData[{

Cell["Invariants", "Subsection",
 CellID->1705354933],

Cell["Consider a differential equation", "Text",
 CellID->1049715760],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    OverscriptBox["y", "."], " ", "=", " ", 
    RowBox[{"f", "(", "y", ")"}]}], ",", " ", 
   RowBox[{
    RowBox[{"y", "(", 
     SubscriptBox["t", "0"], ")"}], " ", "=", " ", 
    SubscriptBox["y", "0"]}], ","}], TraditionalForm]], "NumberedEquation",
 CellTags->"ODESystem",
 CellID->1708949319],

Cell["where y may be a vector or a matrix.", "Text",
 CellID->1601139534],

Cell[TextData[{
 StyleBox["Definition",
  FontWeight->"Bold"],
 ": A non-constant function I(",
 "y",
 ") is called an ",
 StyleBox["invariant",
  FontSlant->"Italic"],
 " of (",
 
 CounterBox["NumberedEquation", "ODESystem"],
 ") if ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      SuperscriptBox["I", "\[Prime]"], "(", "y", ")"}], 
     RowBox[{"f", "(", "y", ")"}]}], " ", "=", " ", "0"}], 
   TraditionalForm]]],
 " for all y."
}], "Text",
 CellID->1999213257],

Cell[TextData[{
 "This implies that every solution ",
 StyleBox["y",
  FontSlant->"Italic"],
 "(",
 StyleBox["t",
  FontSlant->"Italic"],
 ") of (",
 
 CounterBox["NumberedEquation", "ODESystem"],
 ") satisfies I(",
 StyleBox["y",
  FontSlant->"Italic"],
 "(",
 StyleBox["t",
  FontSlant->"Italic"],
 ")) = ",
 Cell[BoxData[
  FormBox[
   RowBox[{"I", "(", 
    SubscriptBox["y", "0"], ")"}], TraditionalForm]]],
 "= Constant."
}], "Text",
 CellID->382477753],

Cell[TextData[{
 "Synonymous with invariant, the terms ",
 StyleBox["first integral",
  FontSlant->"Italic"],
 ", ",
 StyleBox["conserved quantity",
  FontSlant->"Italic"],
 ", or ",
 StyleBox["constant of the motion",
  FontSlant->"Italic"],
 " are also common."
}], "Text",
 CellID->782072918]
}, Open  ]],

Cell[CellGroupData[{

Cell["Manifolds", "Subsection",
 CellID->1496795871],

Cell[TextData[{
 "Given an ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"n", "-", "m"}], ")"}], TraditionalForm]]],
 "-dimensional submanifold of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", "n"], TraditionalForm]]],
 " with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"g", ":", " ", 
    RowBox[{
     SuperscriptBox["\[DoubleStruckCapitalR]", "n"], "\[RightTeeArrow]", 
     RowBox[{
      SuperscriptBox["\[DoubleStruckCapitalR]", "m"], ":"}]}]}], 
   TraditionalForm]]]
}], "Text",
 CellID->1938338644],

Cell[BoxData[
 FormBox[
  RowBox[{"\[ScriptCapitalM]", "=", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"y", ";", 
      RowBox[{
       RowBox[{"g", "(", "y", ")"}], "=", "0"}]}], "}"}], "."}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Submanifold",
 CellID->2090721578],

Cell[TextData[{
 "Given a differential equation (",
 
 CounterBox["NumberedEquation", "ODESystem"],
 ") then ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["y", "0"], "\[Element]", "\[ScriptCapitalM]"}], 
   TraditionalForm]]],
 " implies ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"y", "(", "t", ")"}], "\[Element]", "\[ScriptCapitalM]"}], 
   TraditionalForm]]],
 " for all ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]]],
 ".",
 " ",
 "This is a weaker assumption than invariance and g(y) is called a weak \
invariant (see [",
 ButtonBox["HLW02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2433"],
 "])."
}], "Text",
 CellID->1862310036]
}, Open  ]],

Cell[CellGroupData[{

Cell["Projection algorithm", "Subsection",
 CellID->801349120],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox[
   SubscriptBox[
    FormBox[
     OverscriptBox["y", "\[Tilde]"],
     TraditionalForm], 
    RowBox[{"n", "+", "1"}]], TraditionalForm]]],
 "denote the solution from a one-step numerical integrator. Considering a \
constrained minimization problem leads to the following system (see [",
 ButtonBox["AP91",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#12140"],
 "], [",
 ButtonBox["HW96",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#32197"],
 "] and [",
 ButtonBox["HLW02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2433"],
 "]):"
}], "Text",
 CellID->2003610528],

Cell[BoxData[
 FormBox[GridBox[{
    {
     SubscriptBox["y", 
      RowBox[{"n", "+", "1"}]], "=", 
     RowBox[{
      SubscriptBox[
       OverscriptBox["y", "\[Tilde]"], 
       RowBox[{"n", "+", "1"}]], "+", " ", 
      RowBox[{
       RowBox[{"g", "'"}], 
       SuperscriptBox[
        RowBox[{"(", 
         SubscriptBox["y", 
          RowBox[{"n", "+", "1"}]], ")"}], "T"], "\[Lambda]"}]}]},
    {"0", "=", 
     RowBox[{"g", "(", 
      SubscriptBox["y", 
       RowBox[{"n", "+", "1"}]], ")"}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"ConstrainedMinimum",
 CellID->811982708],

Cell[TextData[{
 "To save work ",
 Cell[BoxData[
  FormBox[
   RowBox[{"g", "(", 
    SubscriptBox["y", 
     RowBox[{"n", "+", "1"}]], ")"}], TraditionalForm]]],
 "is approximated as ",
 Cell[BoxData[
  FormBox[
   RowBox[{"g", "(", 
    SubscriptBox[
     OverscriptBox["y", "\[Tilde]"], 
     RowBox[{"n", "+", "1"}]], ")"}], TraditionalForm]]],
 ". Substituting the first relation into the second relation in (",
 
 CounterBox["NumberedEquation", "ConstrainedMinimum"],
 ") leads to the following simplified Newton scheme for \[Lambda]:"
}], "Text",
 CellID->1860455852],

Cell[BoxData[
 FormBox[GridBox[{
    {
     RowBox[{
      SubscriptBox["\[CapitalDelta]\[Lambda]", "i"], " "}], "=", 
     RowBox[{
      RowBox[{
       RowBox[{"-", 
        SuperscriptBox[
         RowBox[{"(", " ", 
          RowBox[{
           RowBox[{"g", "'"}], 
           RowBox[{"(", 
            SubscriptBox[
             OverscriptBox["y", "\[Tilde]"], 
             RowBox[{"n", "+", "1"}]], ")"}], " ", 
           RowBox[{"g", "'"}], 
           SuperscriptBox[
            RowBox[{"(", 
             SubscriptBox[
              OverscriptBox["y", "\[Tilde]"], 
              RowBox[{"n", "+", "1"}]], ")"}], "T"]}], ")"}], 
         RowBox[{"-", "1"}]]}], 
       RowBox[{"g", "(", 
        RowBox[{
         SubscriptBox[
          OverscriptBox["y", "\[Tilde]"], 
          RowBox[{"n", "+", "1"}]], " ", "+", " ", 
         RowBox[{
          RowBox[{"g", "'"}], 
          SuperscriptBox[
           RowBox[{"(", 
            SubscriptBox[
             OverscriptBox["y", "\[Tilde]"], 
             RowBox[{"n", "+", "1"}]], ")"}], "T"], " ", 
          SubscriptBox["\[Lambda]", "i"]}]}], ")"}]}], " ", ","}]},
    {
     SubscriptBox["\[Lambda]", 
      RowBox[{"i", "+", "1"}]], "=", 
     RowBox[{
      SubscriptBox["\[Lambda]", "i"], "+", " ", 
      SubscriptBox["\[CapitalDelta]\[Lambda]", "i"], " "}]}
   }], TraditionalForm]], "NumberedEquation",
 CellTags->"StandardProjectionAlgorithm",
 CellID->1649127868],

Cell[TextData[{
 "with ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Lambda]", "0"], "=", " ", "0"}], TraditionalForm]]],
 "."
}], "Text",
 CellID->1147204139],

Cell[TextData[{
 "The first increment ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalDelta]\[Lambda]", "0"], TraditionalForm]]],
 "is of size ",
 Cell[BoxData[
  FormBox[
   RowBox[{"O", "(", 
    SubsuperscriptBox["h", "n", 
     RowBox[{"p", "+", "1"}]], ")"}], TraditionalForm]]],
 " so that (",
 
 CounterBox["NumberedEquation", "StandardProjectionAlgorithm"],
 ") usually converges quickly."
}], "Text",
 CellID->1457553951],

Cell["\<\
The added expense of using a higher order integration method can be offset by \
fewer Newton iterations in the projective step.\
\>", "Text",
 CellID->672564483],

Cell[TextData[{
 "For the termination criterion in the method ",
 StyleBox["Projection", "MR"],
 ", the option ",
 StyleBox["IterationSafetyFactor", "MR"],
 " is combined with one Unit in the Last Place in the working precision used \
by ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 "."
}], "Text",
 CellID->1304033423]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
 CellTags->"s:2",
 CellID->1880181156],

Cell["Load some utility packages.", "Text",
 CellID->1794444368],

Cell[BoxData[{
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveProblems`\>\"", "]"}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
  "Needs", "[", "\"\<DifferentialEquations`NDSolveUtilities`\>\"", "]"}], 
  ";"}]}], "Input",
 InitializationCell->True,
 CellLabel->"In[3]:=",
 CellID->1722446242],

Cell[CellGroupData[{

Cell["Linear invariants", "Subsection",
 CellID->940869338],

Cell["Define the a stiff system modeling a chemical reaction.", "Text",
 CellID->594444171],

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<Robertson\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->1183419160],

Cell["This system has a linear invariant.", "Text",
 CellID->697866412],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"invariant", " ", "=", " ", 
  RowBox[{"system", "[", "\"\<Invariants\>\"", "]"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->1256928757],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["Y", "1"], "[", "T", "]"}], "+", 
   RowBox[{
    SubscriptBox["Y", "2"], "[", "T", "]"}], "+", 
   RowBox[{
    SubscriptBox["Y", "3"], "[", "T", "]"}]}], "}"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->936832272]
}, Open  ]],

Cell[TextData[{
 "Linear invariants are generally conserved by numerical integrators (see [",
 ButtonBox["S86",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#29968"],
 "]), including the default ",
 Cell[BoxData[
  ButtonBox["NDSolve",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NDSolve"]], "InlineFormula"],
 " method, as can be observed in a plot of the error in the invariant."
}], "Text",
 CellID->239949637],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"sol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", "system", "]"}]}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{"InvariantErrorPlot", "[", 
  RowBox[{"invariant", ",", "vars", ",", "T", ",", "sol"}], "]"}]}], "Input",
 CellLabel->"In[8]:=",
 CellID->1274449081],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXEtv00AQ3sR5EJW2SDyKIlGlLY8eWlSq3qqStOUhIQpIqEJCrUSoEI0E
bdXHT4AT3OBA6KVQEBUSD1WUGxckBAfElTN/xHjt3dgZr722s27sZKPE3lnP
7sx83ux+njieKa8v3ntYXq8slAuXV8sri5WFtcKl5VWtSkkglFjUPqXjSCur
CBkf/KpoH6MIS6PaNqmpKVhSqL6KaB3e97noJLUCrup30FGITpKjg1K6O+TF
sojreb1RnfrenDStMerH0Dm8TZmekY4y1hpjmwQxnGC3tNdY2vTaNDK2GmjH
oY0VTa5nabxL10KhaHAtpQF+/HiyEHFtrw1TlMD7NkI8DCsSbSfPWDh4Rw4X
vX0X6PzHiChkpOkc2CcOYbOG9F0IA1cfPu/nyIVt7LEHxjVtxRUx4/caid1C
kEj4swTlBBnVXButPIDWpxzqqX6/g37Bpz7tH3/nFATXbByWM79hlcYIEqO1
PsYAanCOgeMYynCOEq3f7PZx89dve3i+je+BHBXRsheNUTGsmrORm/ZZIA9z
jvvVZ3s3hLeurMxYITbW94oTL7Q3SuKNcvXKFjmON70N2MILHlKmSi9J/4q2
wYcHtE+qNuN69DPzauv3VLFanNDf56tLDz7jg6cDxo43EMfG2zqNCOcWcGaJ
siatHFLN1TcKfrWepkRaIt1qmvFG2nrFUb862HKhRJNemR/zbOMAqLWuO3gp
5V95Hg5sC65Y6RBtBYuLXj4PkrIxfugVIvsKmx9FxnK23PBwjg5yx26uzSzT
V8g5nS1mScxdav1VM97XeJWN57Hxd7c0GNWeXPGF8Tv3dcaCnILC423xnvvi
pCmRlkjHQ7Nd8KtbYUNiTnDthL9PimROvNyRSOYkJi42c4J96ytwgwyqxsQC
RdfFtZXlanhf+ylz7FZdWJQgWyI5C2RSii/M9dYNcE/KmOgvMmEyJmPGuD6z
ZeQPb1x7bWQonz75TvT1MdvxZvuPVm/UMsvoqG7HlHMceQTIH1S3ORr7mdE9
uzi9WefZrdm3Wr1RyywTz0w5x5FHgPzJE9Y6ktnVld3SRHWyWN1Y+3L3zvv5
uZ2ve3+NsdPx6+e/5aVdTdYkZpl4aso5jjwCZG+eiho/cdFsFxbQfE2JtMTP
i2b9HYPWdVqXaeS2NTSlBmWvLEaFsqB/YkUkk2XahatGSMxWfMxslsvKMzmf
wzDZLi/jdZBrWyT7JfDoL5pP7FAlE44mE8ZVrT7vNl9TIi3xi4dmvJHGM2UK
hZsRtK4DCDnIMcwINvM3YpGcic3W7H15/DVV5gTbgAm1y/wYJ01zpM3dfme5
zxMplfsfSfb0iH7czKaOAxlmW3uAnPcpD7iedeytka0t4XtJ9Syq4aGZVR0H
Msy69gA571M+5Ql5Hdfc40ffJvWbXi9Mb87e3J6f23n+7AfJqRp+mznWcSDD
HGwPkPM+ZW9+R3m8yjkkeprxRrrGZC0sLIwsIc1LGlt9dW7xHCHkP+l9itn2
r3RVfM6QzX+d/2WNizJDGKgvyYtrmvGeaZuvKfGTSLeaZryRlv/LCG6LZrvg
cwDCzI2KvMuQzaEggp3cvkWyJjoeZQZRMqV21pT4xRlpmIHtBDK8d4XXk5n5
7ASy357MXGQnkL31FI1nM/C8gjNnnnP8UIPHefZONtm+fCpLO/grn9WzH+3j
5q+4UYFJqfmkL5T4D25fk3U=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 320},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->279282938]
}, Open  ]],

Cell[TextData[{
 "Therefore in this example there is no need to use the method ",
 StyleBox["Projection", "MR"],
 "."
}], "Text",
 CellID->1289665398],

Cell[TextData[{
 "Certain numerical methods preserve quadratic invariants exactly (see for \
example [",
 ButtonBox["C87",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#14833"],
 "]). The implicit midpoint rule, or one-stage Gauss implicit Runge-Kutta \
method, is one such method."
}], "Text",
 CellID->835914752]
}, Open  ]],

Cell[CellGroupData[{

Cell["Harmonic oscillator", "Subsection",
 CellID->1871712721],

Cell["Define the Harmonic oscillator.", "Text",
 CellID->1141395007],

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<HarmonicOscillator\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"vars", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}]}], 
  ";"}]}], "Input",
 CellLabel->"In[10]:=",
 CellID->1218392506],

Cell["The Harmonic oscillator has the following invariant.", "Text",
 CellID->506301991],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"invariant", " ", "=", " ", 
  RowBox[{"system", "[", "\"\<Invariants\>\"", "]"}]}]], "Input",
 CellLabel->"In[12]:=",
 CellID->490707230],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", "2"], " ", 
   RowBox[{"(", 
    RowBox[{
     SuperscriptBox[
      RowBox[{
       SubscriptBox["Y", "1"], "[", "T", "]"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}], "2"]}], ")"}]}], 
  "}"}]], "Output",
 ImageSize->{137, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->1047833865]
}, Open  ]],

Cell[TextData[{
 "Solve the system using the method ",
 StyleBox["ExplicitRungeKutta", "MR"],
 ". The error in the invariant grows roughly linearly which is typical \
behavior for a dissipative method applied to a Hamiltonian system."
}], "Text",
 CellID->449539921],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"erksol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}]}], "]"}]}], 
   ";"}], "\n"}], "\n", 
 RowBox[{"InvariantErrorPlot", "[", 
  RowBox[{"invariant", ",", "vars", ",", "T", ",", "erksol"}], 
  "]"}]}], "Input",
 CellLabel->"In[13]:=",
 CellID->205533619],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW01o1FAQzu4muy6W1p9WBKEtCrYsgmJL/0SKiD8UPZZeXYrQWmylloLH
gkdLryreaj0KIiiCJyuop4InQbx4EK/iVXzmJXmdvMl7mU02bRK7C7vJ7Hxv
Zt4385J5afd6fWnm5u360ux0vffKYv3OzOz03d7LC4v2V6WCYRRm7PfVo4Z9
zgzDffPXrP12T/HZgP1ZtGGj9tG0j2P825IYx4wRTz/mHYftI1cXPdlyrJzl
nyaMcu0JO4bC7nBgxEjgmyCmiKwUNd5V3kyljyS9nkO4IcTBqJKlojKKuD6H
ffksKX3Gn18wShG/qmrieR1gcr3h+QmP4ijmW2L+euRQfc2rzk57UY3xRDmS
32sJyW0tuSmZ81loMb2rTPcz3woJYXw0YRki6GP4GmBmOBb3+rTxdGv8/KPV
B5sBmYNOEfYsripNTT6zx1y6+MSzATKH9gtcwzF9+vh9Yf7l61dfAnIjMfnt
8Y8hRlWIjMIrtqVtRiuUg955lmPda1qh5D2JmXo0LW1rVf0P2taqakbLFU6X
JN3Xcd9UZPIu/Ui4TaT1dwlcXw74w3u3w5Hs4/7CSth+0vG7DtyXY8vaRnoK
egZllBEuCzNnmOgAhSf+eYC0WSERmBl5XiKWdibvofnRVNZZNPsncj7+OOKl
HIrevStClyPLO5FuAg17jrJidBcxGnYYZWQtfHQ3Gh1Et+4GWdHWFHVRJdBy
HchoDq4p8l9h8XqyQRTfXwIJsVFIiE+PTCsr42jWvwkkzJpCwqz1yKiR06t5
As3np9b6BJoPhYT56JFpZXHntJOIz28EEvikkMCnHqmvgDR2XjcQF58JJHBB
IYELPTK5efl7xWh9/sFQuxWk9ffhwm94v9QeyT7ul+guGdufQxn9sINzmUM1
offl7C9i7AEsL0vcHh/u/OXJV1ubTOwDsMd20rYyxtAK45F0MHXnL+oq/Enw
yUBU6eL7mLpPX2ZyFb1lunW4jGqAQkLu9Eg5xuxq6Sv3CuLxhdb6CuKRQgKP
emR2mEpKu4r43CCQwCeFBD5lZF53Ww8RU48JJDBFIYEpPTKbnKwjTtYCmV5H
XKwy3cpeR1ysxYwpj9rniMf7BBL4pJDApx4ZNfK8ruBw7RuUg3sEEnJAISEH
eqQcm9s6iZe/z3Jkw+vCA/2fyaL04qrnw0YF2fQsN9uXK32hXURzfXp25mYy
6GST6NHfMbkybzHRo6s8++U20hfVs79ncq3Pb8/a/xeJihfxfhatl3etw/oA
63np62VUtCvOFsrrNIGELFBIYFSPbCbyoDa794SviOUpAgksU0hgWY9Ma9Y7
k924T9R+oBxcI5CQAwoJOdAjs8Nj49pfiLELBBIYo5DAmIzM7gr+g7igkMAF
hQQuslcByWqFsoq47GdhK7qK+KyFeqwiTsPR2eAlrpZ3C6KzSfL5uL+/0MqZ
ej4u29+HtNH/D0Zlv5PJVduj9ZdUz82/PMTk6j/GRN8dtNdBemjk6bg7T1hB
MM+99bRcRkVfpz1MrpfOBtCQZwuNxrLZgDXIoaW0DnK4tT6lNZD1o3fvdwzZ
9Zz2L1/yLstMl4zg3S3tCHMpwxWS+bsv+O2fUfgHcso/Nw==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 328},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->617703590]
}, Open  ]],

Cell[TextData[{
 "This also solves the system using the method ",
 StyleBox["ExplicitRungeKutta", "MR"],
 " but it projects the solution at the end of each step. A plot of the error \
in the invariant shows that it is conserved up to roundoff."
}], "Text",
 CellID->305666802],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"projerksol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<Projection\>\"", ",", 
         RowBox[{"Method", "\[Rule]", "\"\<ExplicitRungeKutta\>\""}], ",", 
         RowBox[{"\"\<Invariants\>\"", "\[Rule]", "invariant"}]}], "}"}]}]}], 
     "]"}]}], ";"}], "\[IndentingNewLine]"}], "\n", 
 RowBox[{"InvariantErrorPlot", "[", 
  RowBox[{"invariant", ",", "vars", ",", "T", ",", "projerksol"}], 
  "]"}]}], "Input",
 CellLabel->"In[15]:=",
 CellID->1017516327],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW8tu00AUndiOk5BSkBKhSJGSii4g3YAoapoghCLEQ0QsEeuoQkoUQVHp
Z8AHILEEfgAJPoANrNiy5keMx49mfDz2xI88ml5Ltuf43rn3zp3JzIkfz0en
41evR6eTo9HOk5PR2/Hk6N3O4+MT+5JeYKwwtvenDWaXLcbcnW8Te3eLWNq3
j5qt1rPPhn3u86u6X89ih56875279pmLNQ8XHSt3+NGY1XLt+XaYxG43VOMw
dCWso4EVLcK7zJsh9ZGn13ugdwA56EmzpEmjSOuzK/SnLvWZvn3hKP34ZaMm
ndd9KzjesH2+R//st1e3xPHIVaPHvKx0y4uqzzvKQaJXHfAW4UyY57NAmV5q
pm9awi8kJuO9nPEsghsWzgHGGsfizk9fPv95cP/jh/c/Q7gK2ARcA1wG3Eqo
v6eQY/1KQv064AZgQ5G7IhfpL198tfUfPfzk5WuGq4BNwDXAZcCthPp7CjnW
ryTUrwNuAC4mGme/f/07fvPtx/e/IVwFbAKuAS4DbiXU31PIsX4loX4dcAOw
apyJueOHA0s1wwW1cMUhabzUF971yusc6+ZJfSHnxcbKoyEp/TJEKR+RznoV
mKWRxWlW8J7BtVib7iTvCO2jKegyH8f+j6zFWkdtE3Axo31su7hi5RE/2ndw
8Qx7XtStMIU+wfb75m57Za+jz2xdUVovSaMqxbbMbYq7cQ/iv3p+NiJGVlS2
wx52V107NmfYtmD960ImdHYxOQ7Nt5R9klLfJJc6c21oPZJfNawkbEW2fua5
3vOquG5m5ShifKLt+eOVMARh9Talec3GSebjDVeVPtIwE7Rx2SJ2Eqx/MdgJ
zbKUfZJG9Q0+n5gC7gAeAG4CvhTZz3hnfwq4A3gAuAl4K7KleC98CrgDeAC4
CTja0/r0ZVqpuA47xVzvCYm28r4nFKybO9vKIfYw39LZqngWPy6OZ/GxsG0R
vwrWJ361brFunpSyv75S6hviHqmsC3nJ7U5PoI4T4fI4CD2HIiayBCnNt5R9
klLfJJf6b6+4W97vxiz6adOi347JJ/4wJ5GugRlZicZYqP3xrdlW2i8pNeJX
37m5SQYfWRkC8hNdmcddaW/FszSfhxjWJr/zS7Ptec7+MwfP/wVGWyEfAsYn
LnVFJPN/29BWyIeA8YmMKpL5vxZoK+RDwPjEJjqS5X1rtL6eV/112nnHlOnV
ZHr2pS4r/AefGTha\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 328},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->219972584]
}, Open  ]],

Cell["\<\
Since the system is Hamiltonian (the invariant is the Hamiltonian), a \
symplectic integrator performs well on this problem giving a small bounded \
error.\
\>", "Text",
 CellID->1954794670],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"projerksol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<SymplecticPartitionedRungeKutta\>\"", ",", 
         RowBox[{"\"\<DifferenceOrder\>\"", "\[Rule]", "8"}], ",", 
         RowBox[{"\"\<PositionVariables\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{
            SubscriptBox["Y", "1"], "[", "T", "]"}], "}"}]}]}], "}"}]}], ",", 
      
      RowBox[{"StartingStepSize", "\[Rule]", 
       RowBox[{"1", "/", "5"}]}]}], "]"}]}], ";"}], 
  "\[IndentingNewLine]"}], "\n", 
 RowBox[{"InvariantErrorPlot", "[", 
  RowBox[{"invariant", ",", "vars", ",", "T", ",", "projerksol"}], 
  "]"}]}], "Input",
 CellLabel->"In[17]:=",
 CellID->17236819],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztm0uIHFUUhqufM5hmIkkIEkQakRicBBk08xCGLIYoIiIuJllEwTYYZkAT
GWcRshAX4kIihICgjC7EFy7ERaIiLlREVFwoLsIsxEVw5UJchCxmU/atR/66
f9W5p7qrq7tHZqC7uup+95zz/3Wr+t6e7sc76yvPvdBZXz3daT+y1nlxZfX0
S+2Hz611D9UqnldZ6T6W7vC6r33PCx/mb7X7CF/yq/nuc9ULt/XudsEcrcX9
fG8ual+ItrPdrWmuRvuNIIqhvDp6hfHiOF5G3NlUj3SMNFOlKFUhe1a2emaO
QWZ9iLij5MF8pkvVzCr6zTmbOJ+1zJz968s+y/Ho4FHTX9ZkxGqGvuR4jcej
2db85Hg0qDzms149GFV11JyoYC+ZtUb7LWV/V8n8du9v/KzsOD1Up4/4iSvE
4fj8gPdRwWGf7wH1Ma4lvD99+MGvxxbffuPi98HhvXRsvxKvYZpqJ5c/6vLH
l97xTlH/q8E+2vfEfXLX9/NP18+dvfLlF5tRbMQ6QblOEa/Vnsz1DPU9QbnW
fW2UfUURalTdWV+6I3xDuWSS3e1QTq0ncvyukIgpk+bpLqro9hw0qpDpSXFs
ojKtN/KYgwcVGnHdtHnaQ9XdmYNGNXloVGPTMXx/9Do7xoyY4QxV/qNComqN
RMUy6a7tCtV2QSFRm0aiNpl011ak9QfStaKQ0KWR0CWT7tqWqbbN1Ghbppqu
iSNvmWra7LOmoq33kKbbctDQl4eGRjfdr4opUmAOthXavttptH23k+nBnqm4
sUX62r7rbtYifXc7M7ZIn5subxT22/o8efOdQsIXjYQnMlmerk9J1ysKCV0a
CV02WfS9srzWX8iLZxUSXmgkvJDJ8nQ9Qbr+VEjoAtmIzlYwf7bm7Lx2qvr2
J3X7M3PAEeSYJDK5GjDtzVRu/ixnr9MXjs9rhcaA4/dXfyPanfbjNVkzcjLo
37hFR1H1qpvJcyfUl1bTpNbdap4J2ud2d76JSOWUb3+yZrb1rOq99GrPjngo
EJ+sOHxmZebpPt++Rup2FOhOnovMGLh6+o+BqwMx7o16sYtuV4d3l9lHDpqD
BxTanh1ptD07kunBasz3ftUk9YecGZqkPg8N9W56/OZO7tY1cu5rhYRrGgnH
ZHIUY0WO+DF58ZpCwguNhBcyOZ7jw936Gzn2tELCMY2EYzI5XqPnUfLiL4WE
FxoJL2RyPMfHdmktd1V0wE+/N+9TaPu9WaPt92aZHqXjFXLhSA4aLuSh4QLo
ePYdzIYLrZxshTzbTs6F47xFVjMcn2erRVdLg6k/vVriuMH8uMCqKQwQ/2XN
2WVNU2oW7sFjgvftfPHKcLefb83UW/R8q5bzvn1VfW7372E9GEbCFSdH0nSE
kXA1IlK8ioqvNff5HN796T1y8aJCwiWNhAsyOVhd5b6XuVuvkY8nFRI+aiR8
lMlRjZ4lUv23QkK1RkK1TG6n8fH/a23TuTcHpxTanttptD23k+nBatRHyRYp
f0CMvkWqNRKKZXI8R8PL5MlnCglPNBKeyGR5ujZI1+XUWNkgPZfEcbNBei73
WdOoWv8gL55USHiikfBEJgeny55ZJ2eHwb4XzV9Tc/e6714nLZI//97KmDXH
9yYofpSl9/XTIrmt5HXMjHtfWy3S+RuWZvuMZs3w5fNYZA2WXvdpq4uWmq3Y
WixsDv/i/2Xt8ke3Lsvo08NazN27t3rLXHMd9O1r3RycVGh79qPR9uxHpouo
uEEqFhQSCjQS1dtkuXPwV0nPJwoJPRoJPTLpru1Nqu0thURtGonaZLKIq+7W
66TrMYWELo2ELpmco+w3U6NtjrLeEEfeHGW9KWYtz0136zSpNQfl35ZMk/I8
NNR/S7mOiT3/oTwy+TrF5G+saz2R432FlL95/67Y8xLl4G/DHhd7zhD5FNXA
35rVIoG8SpHsnqbjDBFb4vge3q9Nxjfzzu+rivXfcXo0TuO3ml7lPwu4/pM=
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 323},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[18]=",
 CellID->9956314]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Perturbed Kepler problem", "Subsection",
 CellID->479875506],

Cell["\<\
This loads a Hamiltonian system known as the perturbed Kepler problem, sets \
the integration interval and the step size to take as well as defining the \
position variables in the Hamiltonian formalism.\
\>", "Text",
 CellID->1321270242],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"system", " ", "=", " ", 
   RowBox[{"GetNDSolveProblem", "[", "\"\<PerturbedKepler\>\"", "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"time", " ", "=", " ", 
   RowBox[{"system", "[", "\"\<TimeData\>\"", "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"step", " ", "=", " ", 
   RowBox[{"3", "/", "100"}]}], ";"}], "\n", 
 RowBox[{"pvars", " ", "=", " ", 
  RowBox[{"Take", "[", 
   RowBox[{
    RowBox[{"system", "[", "\"\<DependentVariables\>\"", "]"}], ",", " ", 
    "2"}], "]"}]}]}], "Input",
 CellLabel->"In[19]:=",
 CellID->566336639],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SubscriptBox["Y", "1"], "[", "T", "]"}], ",", 
   RowBox[{
    SubscriptBox["Y", "2"], "[", "T", "]"}]}], "}"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->776728042]
}, Open  ]],

Cell[TextData[{
 "The system has two invariants which ar defined as ",
 StyleBox["H", "MR"],
 " and ",
 StyleBox["L", "MR"],
 "."
}], "Text",
 CellID->896232836],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{"H", ",", "L"}], "}"}], " ", "=", " ", 
  RowBox[{"system", "[", "\"\<Invariants\>\"", "]"}]}]], "Input",
 CellLabel->"In[23]:=",
 CellID->1507205977],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"-", 
     FractionBox["1", 
      RowBox[{"400", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{
          SuperscriptBox[
           RowBox[{
            SubscriptBox["Y", "1"], "[", "T", "]"}], "2"], "+", 
          SuperscriptBox[
           RowBox[{
            SubscriptBox["Y", "2"], "[", "T", "]"}], "2"]}], ")"}], 
        RowBox[{"3", "/", "2"}]]}]]}], "-", 
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox[
        RowBox[{
         SubscriptBox["Y", "1"], "[", "T", "]"}], "2"], "+", 
       SuperscriptBox[
        RowBox[{
         SubscriptBox["Y", "2"], "[", "T", "]"}], "2"]}]]], "+", 
    RowBox[{
     FractionBox["1", "2"], " ", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox[
        RowBox[{
         SubscriptBox["Y", "3"], "[", "T", "]"}], "2"], "+", 
       SuperscriptBox[
        RowBox[{
         SubscriptBox["Y", "4"], "[", "T", "]"}], "2"]}], ")"}]}]}], ",", 
   RowBox[{
    RowBox[{
     RowBox[{"-", 
      RowBox[{
       SubscriptBox["Y", "2"], "[", "T", "]"}]}], " ", 
     RowBox[{
      SubscriptBox["Y", "3"], "[", "T", "]"}]}], "+", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "1"], "[", "T", "]"}], " ", 
     RowBox[{
      SubscriptBox["Y", "4"], "[", "T", "]"}]}]}]}], "}"}]], "Output",
 ImageSize->{432, 64},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->285552196]
}, Open  ]],

Cell[TextData[{
 "An experiment now illustrates the importance of using all the available \
invariants in the projective process (see [",
 ButtonBox["HLW02",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveReferences#2433"],
 "] ). Consider the solutions obtained using:"
}], "Text",
 CellID->2079651996],

Cell[TextData[{
 "The method ",
 StyleBox["ExplicitEuler", "MR"],
 "."
}], "BulletedText",
 CellID->57588741],

Cell[TextData[{
 "The method ",
 StyleBox["Projection", "MR"],
 " with ",
 StyleBox["ExplicitEuler", "MR"],
 ", projecting onto the invariant L."
}], "BulletedText",
 CellID->1978209155],

Cell[TextData[{
 "The method ",
 StyleBox["Projection", "MR"],
 " with ",
 StyleBox["ExplicitEuler", "MR"],
 ", projecting onto the invariant H."
}], "BulletedText",
 CellID->644870249],

Cell[CellGroupData[{

Cell[TextData[{
 "The method ",
 StyleBox["Projection", "MR"],
 " with ",
 StyleBox["ExplicitEuler", "MR"],
 ", projecting onto both the invariants H and L."
}], "BulletedText",
 CellID->297538379],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"sol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"Method", "\[Rule]", "\"\<ExplicitEuler\>\""}], ",", " ", 
      RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"pvars", " ", "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", " ", 
   RowBox[{"Evaluate", "[", "time", "]"}]}], "]"}]}], "Input",
 CellLabel->"In[24]:=",
 CellID->1105174588],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->133081654]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"sol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<Projection\>\"", ",", " ", 
         RowBox[{"Method", "->", "\"\<ExplicitEuler\>\""}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"\"\<Invariants\>\"", "\[Rule]", 
          RowBox[{"{", "H", "}"}]}]}], "}"}]}], ",", " ", 
      RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"pvars", " ", "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", " ", 
   RowBox[{"Evaluate", "[", "time", "]"}]}], "]"}]}], "Input",
 CellLabel->"In[26]:=",
 CellID->1587203777],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 300},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[27]=",
 CellID->460416710]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"sol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<Projection\>\"", ",", " ", 
         RowBox[{"Method", "->", "\"\<ExplicitEuler\>\""}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"\"\<Invariants\>\"", "\[Rule]", 
          RowBox[{"{", "L", "}"}]}]}], "}"}]}], ",", " ", 
      RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"pvars", " ", "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", " ", 
   RowBox[{"Evaluate", "[", "time", "]"}]}], "]"}]}], "Input",
 CellLabel->"In[28]:=",
 CellID->639672509],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 324},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[29]=",
 CellID->425620513]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"sol", " ", "=", " ", 
    RowBox[{"NDSolve", "[", 
     RowBox[{"system", ",", " ", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<Projection\>\"", ",", " ", 
         RowBox[{"Method", "->", "\"\<ExplicitEuler\>\""}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"\"\<Invariants\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"H", ",", "L"}], "}"}]}]}], "}"}]}], ",", " ", 
      RowBox[{"StartingStepSize", "\[Rule]", "step"}]}], "]"}]}], ";"}], 
  "\n"}], "\[IndentingNewLine]", 
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"Evaluate", "[", 
    RowBox[{"pvars", " ", "/.", " ", 
     RowBox[{"First", "[", "sol", "]"}]}], "]"}], ",", " ", 
   RowBox[{"Evaluate", "[", "time", "]"}]}], "]"}]}], "Input",
 CellLabel->"In[30]:=",
 CellID->1011998199],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 361},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[31]=",
 CellID->12797868]
}, Open  ]],

Cell[TextData[{
 "It can be observed that only the solution with projection onto both \
invariants gives the correct qualitative behavior -",
 " ",
 "for comparison, results using an efficient symplectic solver can be found \
in \"",
 ButtonBox["SymplecticPartitionedRungeKutta Method for NDSolve",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveSPRK"],
 "\"."
}], "Text",
 CellID->1071822366]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Lotka Volterra", "Subsection",
 CellID->2021245701],

Cell[TextData[{
 "An example of constraint projection for the Lotka-Volterra system is given \
within \"",
 ButtonBox["Numerical methods for solving the Lotka-Volterra equations",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveLotkaVolterra"],
 "\"."
}], "Text",
 CellID->1595514422]
}, Open  ]],

Cell[CellGroupData[{

Cell["Euler's equations", "Subsection",
 CellID->1277355985],

Cell[TextData[{
 "An example of constraint projection for Euler's equations is given within \
\"",
 ButtonBox["Rigid body solvers",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NDSolveRigidBody"],
 "\"."
}], "Text",
 CellID->1684358676]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
 CellTags->"s:3",
 CellID->1901181224],

Cell[BoxData[
 FormBox[GridBox[{
    {Cell[TextData[StyleBox["option name", "TableHeader"]]], Cell[TextData[
     StyleBox["default value", "TableHeader"]]], " "},
    {
     StyleBox["Invariants", "MR"], 
     StyleBox["None", "MR"], 
     RowBox[{
     "Specifies", " ", "the", " ", "invariants", " ", "of", " ", "the", " ", 
      "differential", " ", 
      RowBox[{"system", "."}]}]},
    {
     StyleBox["IterationSafetyFactor", "MR"], 
     StyleBox[
      FractionBox[
       StyleBox["1", "MR"], "10"], "MR"], 
     RowBox[{
     "Specifies", " ", "the", " ", "safety", " ", "factor", " ", "to", " ", 
      "use", " ", "in", " ", "the", " ", "iterative", " ", "solution", " ", 
      "of", " ", "the", " ", 
      RowBox[{"invariants", "."}]}]},
    {
     StyleBox["MaxIterations", "MR"], 
     StyleBox["Automatic", "MR"], 
     RowBox[{
     "Specifies", " ", "the", " ", "maximum", " ", "number", " ", "of", " ", 
      "iterations", " ", "to", " ", "use", " ", "in", " ", "the", " ", 
      "iterative", " ", "solution", " ", "of", " ", "the", " ", 
      RowBox[{"invariants", "."}]}]},
    {
     StyleBox["Method", "MR"], 
     StyleBox["StiffnessSwitching", "MR"], 
     RowBox[{
     "Specifies", " ", "the", " ", "method", " ", "to", " ", "use", " ", 
      "for", " ", "integrating", " ", "the", " ", "differential", " ", 
      "system", " ", 
      RowBox[{"numerically", "."}]}]}
   }], TextForm]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellID->147388665],

Cell[TextData[{
 "Options of the method ",
 StyleBox["Projection", "MR"],
 ". "
}], "Caption",
 CellID->1060443855]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Projection Method for NDSolve - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox[
        "Advanced Numerical Differential Equation Solving in Mathematica", 
         "OverviewNavText"], BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NDSolveOverview"]], "Text", FontFamily -> "Verdana"],
     "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveOrthogonalProjection"],
         "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> "paclet:tutorial/NDSolveStiffnessSwitching"]}],
       "Text", FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 40, 16.6345438}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> "Mathematica Tutorial", 
    "language" -> "en", "paclet" -> "Mathematica", "status" -> "None", 
    "summary" -> 
    "When a differential system has a certain structure, it is advantageous \
if a numerical integration method preserves the structure. In certain \
situations it is useful to solve differential equations in which solutions \
are constrained. Projection methods work by taking a time step with a \
numerical integration method and then projecting the approximate solution \
onto the manifold on which the true solution evolves. NDSolve includes a \
differential algebraic solver which may be appropriate and is described in \
more detail within Numerical Solution of Differential-Algebraic Equations. \
Sometimes the form of the equations may not be reduced to the form required \
by a DAE solver. Furthermore so-called index reduction techniques can destroy \
certain structural properties, such as symplecticity, that the differential \
system may possess (see [HW96] and [HLW02]). An example that illustrates this \
can be found in the documentation for DAEs.", "synonyms" -> {}, "title" -> 
    "Projection Method for NDSolve", "type" -> "Tutorial", "uri" -> 
    "tutorial/NDSolveProjection"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "c:1"->{
  Cell[728, 29, 84, 2, 70, "Title",
   CellTags->"c:1",
   CellID->947483451]},
 "s:1"->{
  Cell[837, 35, 70, 2, 70, "Section",
   CellTags->"s:1",
   CellID->1608456566]},
 "ODESystem"->{
  Cell[3397, 115, 346, 11, 70, "NumberedEquation",
   CellTags->"ODESystem",
   CellID->1708949319]},
 "Submanifold"->{
  Cell[5710, 222, 285, 10, 70, "NumberedEquation",
   CellTags->"Submanifold",
   CellID->2090721578]},
 "ConstrainedMinimum"->{
  Cell[7489, 296, 605, 21, 70, "NumberedEquation",
   CellTags->"ConstrainedMinimum",
   CellID->811982708]},
 "StandardProjectionAlgorithm"->{
  Cell[8674, 340, 1442, 44, 70, "NumberedEquation",
   CellTags->"StandardProjectionAlgorithm",
   CellID->1649127868]},
 "s:2"->{
  Cell[11354, 439, 66, 2, 70, "Section",
   CellTags->"s:2",
   CellID->1880181156]},
 "s:3"->{
  Cell[124682, 2666, 72, 2, 70, "Section",
   CellTags->"s:3",
   CellID->1901181224]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"c:1", 129563, 2786},
 {"s:1", 129653, 2790},
 {"ODESystem", 129752, 2794},
 {"Submanifold", 129872, 2798},
 {"ConstrainedMinimum", 130001, 2802},
 {"StandardProjectionAlgorithm", 130145, 2806},
 {"s:2", 130276, 2810},
 {"s:3", 130372, 2814}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 103, 2, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[728, 29, 84, 2, 70, "Title",
 CellTags->"c:1",
 CellID->947483451],
Cell[CellGroupData[{
Cell[837, 35, 70, 2, 70, "Section",
 CellTags->"s:1",
 CellID->1608456566],
Cell[910, 39, 468, 8, 70, "Text",
 CellID->1904480722],
Cell[1381, 49, 438, 12, 70, "Text",
 CellID->662484585],
Cell[1822, 63, 608, 15, 70, "Text",
 CellID->1676766389],
Cell[2433, 80, 276, 7, 70, "Text",
 CellID->1689566873],
Cell[2712, 89, 532, 16, 70, "Text",
 CellID->17520324],
Cell[CellGroupData[{
Cell[3269, 109, 53, 1, 70, "Subsection",
 CellID->1705354933],
Cell[3325, 112, 69, 1, 70, "Text",
 CellID->1049715760],
Cell[3397, 115, 346, 11, 70, "NumberedEquation",
 CellTags->"ODESystem",
 CellID->1708949319],
Cell[3746, 128, 73, 1, 70, "Text",
 CellID->1601139534],
Cell[3822, 131, 489, 22, 70, "Text",
 CellID->1999213257],
Cell[4314, 155, 459, 23, 70, "Text",
 CellID->382477753],
Cell[4776, 180, 295, 12, 70, "Text",
 CellID->782072918]
}, Open  ]],
Cell[CellGroupData[{
Cell[5108, 197, 52, 1, 70, "Subsection",
 CellID->1496795871],
Cell[5163, 200, 544, 20, 70, "Text",
 CellID->1938338644],
Cell[5710, 222, 285, 10, 70, "NumberedEquation",
 CellTags->"Submanifold",
 CellID->2090721578],
Cell[5998, 234, 692, 28, 70, "Text",
 CellID->1862310036]
}, Open  ]],
Cell[CellGroupData[{
Cell[6727, 267, 62, 1, 70, "Subsection",
 CellID->801349120],
Cell[6792, 270, 694, 24, 70, "Text",
 CellID->2003610528],
Cell[7489, 296, 605, 21, 70, "NumberedEquation",
 CellTags->"ConstrainedMinimum",
 CellID->811982708],
Cell[8097, 319, 574, 19, 70, "Text",
 CellID->1860455852],
Cell[8674, 340, 1442, 44, 70, "NumberedEquation",
 CellTags->"StandardProjectionAlgorithm",
 CellID->1649127868],
Cell[10119, 386, 174, 8, 70, "Text",
 CellID->1147204139],
Cell[10296, 396, 437, 16, 70, "Text",
 CellID->1457553951],
Cell[10736, 414, 171, 4, 70, "Text",
 CellID->672564483],
Cell[10910, 420, 395, 13, 70, "Text",
 CellID->1304033423]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[11354, 439, 66, 2, 70, "Section",
 CellTags->"s:2",
 CellID->1880181156],
Cell[11423, 443, 64, 1, 70, "Text",
 CellID->1794444368],
Cell[11490, 446, 312, 11, 70, "Input",
 InitializationCell->True,
 CellID->1722446242],
Cell[CellGroupData[{
Cell[11827, 461, 59, 1, 70, "Subsection",
 CellID->940869338],
Cell[11889, 464, 91, 1, 70, "Text",
 CellID->594444171],
Cell[11983, 467, 318, 10, 70, "Input",
 CellID->1183419160],
Cell[12304, 479, 71, 1, 70, "Text",
 CellID->697866412],
Cell[CellGroupData[{
Cell[12400, 484, 161, 4, 70, "Input",
 CellID->1256928757],
Cell[12564, 490, 373, 13, 36, "Output",
 CellID->936832272]
}, Open  ]],
Cell[12952, 506, 442, 12, 70, "Text",
 CellID->239949637],
Cell[CellGroupData[{
Cell[13419, 522, 317, 9, 65, "Input",
 CellID->1274449081],
Cell[13739, 533, 1945, 36, 341, "Output",
 Evaluatable->False,
 CellID->279282938]
}, Open  ]],
Cell[15699, 572, 150, 5, 70, "Text",
 CellID->1289665398],
Cell[15852, 579, 338, 9, 70, "Text",
 CellID->835914752]
}, Open  ]],
Cell[CellGroupData[{
Cell[16227, 593, 62, 1, 70, "Subsection",
 CellID->1871712721],
Cell[16292, 596, 68, 1, 70, "Text",
 CellID->1141395007],
Cell[16363, 599, 328, 10, 70, "Input",
 CellID->1218392506],
Cell[16694, 611, 88, 1, 70, "Text",
 CellID->506301991],
Cell[CellGroupData[{
Cell[16807, 616, 161, 4, 70, "Input",
 CellID->490707230],
Cell[16971, 622, 452, 17, 51, "Output",
 CellID->1047833865]
}, Open  ]],
Cell[17438, 642, 266, 6, 70, "Text",
 CellID->449539921],
Cell[CellGroupData[{
Cell[17729, 652, 404, 12, 65, "Input",
 CellID->205533619],
Cell[18136, 666, 1848, 34, 349, "Output",
 Evaluatable->False,
 CellID->617703590]
}, Open  ]],
Cell[19999, 703, 276, 6, 70, "Text",
 CellID->305666802],
Cell[CellGroupData[{
Cell[20300, 713, 613, 16, 83, "Input",
 CellID->1017516327],
Cell[20916, 731, 1450, 28, 349, "Output",
 Evaluatable->False,
 CellID->219972584]
}, Open  ]],
Cell[22381, 762, 200, 5, 70, "Text",
 CellID->1954794670],
Cell[CellGroupData[{
Cell[22606, 771, 813, 22, 101, "Input",
 CellID->17236819],
Cell[23422, 795, 2225, 41, 344, "Output",
 Evaluatable->False,
 CellID->9956314]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[25696, 842, 66, 1, 70, "Subsection",
 CellID->479875506],
Cell[25765, 845, 248, 5, 70, "Text",
 CellID->1321270242],
Cell[CellGroupData[{
Cell[26038, 854, 573, 17, 70, "Input",
 CellID->566336639],
Cell[26614, 873, 310, 11, 36, "Output",
 CellID->776728042]
}, Open  ]],
Cell[26939, 887, 161, 7, 70, "Text",
 CellID->896232836],
Cell[CellGroupData[{
Cell[27125, 898, 201, 6, 70, "Input",
 CellID->1507205977],
Cell[27329, 906, 1491, 52, 85, "Output",
 CellID->285552196]
}, Open  ]],
Cell[28835, 961, 315, 8, 70, "Text",
 CellID->2079651996],
Cell[29153, 971, 109, 5, 70, "BulletedText",
 CellID->57588741],
Cell[29265, 978, 186, 7, 70, "BulletedText",
 CellID->1978209155],
Cell[29454, 987, 185, 7, 70, "BulletedText",
 CellID->644870249],
Cell[CellGroupData[{
Cell[29664, 998, 197, 7, 70, "BulletedText",
 CellID->297538379],
Cell[CellGroupData[{
Cell[29886, 1009, 593, 16, 65, "Input",
 CellID->1105174588],
Cell[30482, 1027, 2596, 47, 80, "Output",
 Evaluatable->False,
 CellID->133081654]
}, Open  ]],
Cell[CellGroupData[{
Cell[33115, 1079, 835, 22, 83, "Input",
 CellID->1587203777],
Cell[33953, 1103, 38233, 631, 321, "Output",
 Evaluatable->False,
 CellID->460416710]
}, Open  ]],
Cell[CellGroupData[{
Cell[72223, 1739, 834, 22, 83, "Input",
 CellID->639672509],
Cell[73060, 1763, 9636, 162, 345, "Output",
 Evaluatable->False,
 CellID->425620513]
}, Open  ]],
Cell[CellGroupData[{
Cell[82733, 1930, 867, 23, 83, "Input",
 CellID->1011998199],
Cell[83603, 1955, 39863, 658, 382, "Output",
 Evaluatable->False,
 CellID->12797868]
}, Open  ]],
Cell[123481, 2616, 405, 11, 70, "Text",
 CellID->1071822366]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[123935, 2633, 57, 1, 70, "Subsection",
 CellID->2021245701],
Cell[123995, 2636, 295, 8, 70, "Text",
 CellID->1595514422]
}, Open  ]],
Cell[CellGroupData[{
Cell[124327, 2649, 60, 1, 70, "Subsection",
 CellID->1277355985],
Cell[124390, 2652, 243, 8, 70, "Text",
 CellID->1684358676]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[124682, 2666, 72, 2, 70, "Section",
 CellTags->"s:3",
 CellID->1901181224],
Cell[124757, 2670, 1641, 42, 70, "DefinitionBox3Col",
 CellID->147388665],
Cell[126401, 2714, 115, 5, 70, "Caption",
 CellID->1060443855]
}, Open  ]]
}, Open  ]],
Cell[126543, 2723, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

