(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[   1321152,      35823]
NotebookOptionsPosition[   1204282,      31964]
NotebookOutlinePosition[   1223711,      32480]
CellTagsIndexPosition[   1220109,      32396]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Advanced Numerical Integration in Mathematica" :> 
          Documentation`HelpLookup["paclet:tutorial/NIntegrateOverview"], 
          "NIntegrate Introduction" :> 
          Documentation`HelpLookup["paclet:tutorial/NIntegrateIntroduction"], 
          "NIntegrate Integration Rules" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NIntegrateIntegrationRules"], 
          "NIntegrate References" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NIntegrateReferences"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Advanced Numerical Integration in Mathematica\"\>", 
       2->"\<\"NIntegrate Introduction\"\>", 
       3->"\<\"NIntegrate Integration Rules\"\>", 
       4->"\<\"NIntegrate References\"\>"}, "\<\"related tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numerical Evaluation & Precision" :> 
          Documentation`HelpLookup[
           "paclet:guide/NumericalEvaluationAndPrecision"], "Calculus" :> 
          Documentation`HelpLookup["paclet:guide/Calculus"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Numerical Evaluation & Precision\"\>", 
       2->"\<\"Calculus\"\>"}, "\<\"more about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Integration Strategies", "Title",
 CellTags->"IntegrationStrategiesChapter",
 CellID->335176949],

Cell[CellGroupData[{

Cell["Introduction", "Section",
 CellID->235955634],

Cell["\<\
An integration strategy is an algorithm that attempts to compute integral \
estimates that satisfy user-specified precision or accuracy goals.\
\>", "Text",
 CellID->865649185],

Cell["\<\
An integration strategy prescribes how to manage and create new elements of a \
set of disjoint subregions of the initial integral region. Each subregion \
might have its own integrand and integration rule associated with it. The \
integral estimate is the sum of the integral estimates of all subregions. \
Integration strategies use integration rules to compute the subregion \
integral estimates. An integration rule samples the integrand with a set of \
points, called abscissas (or sampling points).\
\>", "Text",
 CellID->152119648],

Cell["\<\
To improve an integral estimate the integrand should be sampled at additional \
points. There are two principal approaches: (i) adaptive strategies try to \
identify the problematic integration areas and concentrate the computational \
efforts (i.e. sampling points) on them; (ii) non-adaptive strategies increase \
the number of sampling points over the whole region in order to compute a \
higher degree integration rule estimate that reuses the integrand evaluations \
of the former integral estimate.\
\>", "Text",
 CellID->8231484],

Cell["\<\
Both approaches can use symbolic preprocessing and variable transformation or \
sequence summation acceleration to achieve faster convergence. \
\>", "Text",
 CellID->276385096],

Cell[TextData[{
 "In the following integration ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s symbolic piecewise preprocessor recognizes the integrand as a piecewise \
function; and the integration is done over regions for which ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "\[GreaterEqual]", "1"}], TraditionalForm]], "InlineMath"],
 " with the integrand ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", 
    SqrtBox[
     RowBox[{"x", "-", "1"}]]], TraditionalForm]], "InlineMath"],
 " and regions for which ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "\[LessEqual]", "1"}], TraditionalForm]], "InlineMath"],
 " with the integrand ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", 
    SqrtBox[
     RowBox[{"1", "-", "x"}]]], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->175150228],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    SqrtBox[
     RowBox[{"Abs", "[", 
      RowBox[{"x", "-", "1"}], "]"}]]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[31]:=",
 CellID->234906314],

Cell[BoxData["3.9999999999999725`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[31]=",
 CellID->14379347]
}, Open  ]],

Cell[TextData[{
 "Here is a plot of all sampling points used in the integration. The \
integrand is sampled at the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " coordinates in the order of the ",
 Cell[BoxData[
  StyleBox["y", "TI"]], "InlineFormula"],
 " coordinates (in the plot). It can be seen that the sampling points are \
concentrated near the singularity point ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 ". The patterns formed by the sampling points at the upper part of the plot \
differ from the patterns of the lower part of the plot because a singularity \
handler is applied. "
}], "MathCaption",
 CellID->13516241],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"points", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        SqrtBox[
         RowBox[{"Abs", "[", 
          RowBox[{"x", "-", "1"}], "]"}]]], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"EvaluationMonitor", ":>", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"PointSize", "[", "0.006", "]"}], ",", 
     RowBox[{"Point", "/@", 
      RowBox[{"N", "@", 
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"points", ",", 
          RowBox[{"Range", "[", 
           RowBox[{"Length", "[", "points", "]"}], "]"}]}], "}"}], 
        "]"}]}]}]}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"AspectRatio", "->", "1"}], ",", 
   RowBox[{"Axes", "->", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[10]:=",
 CellID->708501024],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 358},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[11]=",
 CellID->838157734]
}, Open  ]],

Cell[TextData[{
 "The section ",
 ButtonBox["Adaptive Strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#155948475"],
 " gives general description of the adaptive strategies. ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s default (main) strategy is global adaptive and it is explained in the \
section ",
 ButtonBox["Global Adaptive Strategy",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#137878679"],
 ". Complementary to it is ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s local adaptive strategy explained in the section ",
 ButtonBox["Local Adaptive Strategy",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#19475523"],
 ". Both adaptive strategies use singularity handling mechanisms explained in \
the section ",
 ButtonBox["Singularity Handling",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#122144792"],
 ". "
}], "Text",
 CellID->28867535],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s Monte Carlo strategies are explained in the sections ",
 ButtonBox["Crude Monte Carlo and Quasi Monte Carlo Strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#20795380"],
 " and ",
 ButtonBox["Global Adaptive Monte Carlo and Quasi Monte Carlo Strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#65285686"],
 "."
}], "Text",
 CellID->870801276],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s strategies for special type of integrals (or integrands) are explained \
in the corresponding sections: ",
 ButtonBox["Duffy's coordinates strategy",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#66551248"],
 ", ",
 ButtonBox["Oscillatory strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#775797877"],
 ", ",
 ButtonBox["Cauchy principal value integration",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#295594003"],
 "."
}], "Text",
 CellID->548953341],

Cell[TextData[{
 "Here is a table with links to descriptions of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s built-in integration strategies."
}], "MathCaption",
 CellID->68855979],

Cell[BoxData[GridBox[{
   {Cell["strategies", "TableText"], Cell["deterministic", "TableText"], Cell[
    "Monte Carlo", "TableText"]},
   {Cell["adaptive", "TableText"], 
    ButtonBox["\"\<GlobalAdaptive\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#137878679"],
     
    ButtonBox["\"\<AdaptiveMonteCarlo\>\"",
     BaseStyle->"Link",
     ButtonData->
      "paclet:tutorial/NIntegrateIntegrationStrategies#65285686"]},
   {"\[SpanFromAbove]", 
    ButtonBox["\"\<LocalAdaptive\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#19475523"], 
    
    ButtonBox["\"\<AdaptiveQuasiMonteCarlo\>\"",
     BaseStyle->"Link",
     ButtonData->
      "paclet:tutorial/NIntegrateIntegrationStrategies#65285686"]},
   {Cell["non-adaptive", "TableText"], 
    ButtonBox["\"\<DoubleExponential\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#526196975"],
     
    ButtonBox["\"\<MonteCarlo\>\"",
     BaseStyle->"Link",
     ButtonData->
      "paclet:tutorial/NIntegrateIntegrationStrategies#20795380"]},
   {"\[SpanFromAbove]", 
    ButtonBox["\"\<Trapezoidal\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#403370786"],
     ""},
   {Cell["specialized", "TableText"], 
    ButtonBox["\"\<DuffyCoordinates\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#66551248"], 
    ""},
   {"\[SpanFromAbove]", 
    ButtonBox["\"\<Oscillatory\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#775797877"],
     ""},
   {"\[SpanFromAbove]", 
    ButtonBox["\"\<PrincipalValue\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#295594003"],
     ""}
  },
  GridBoxDividers->{
   "Columns" -> {{True}}, 
    "Rows" -> {
     True, True, False, True, False, True, False, False, True}}]], "Text",
 CellID->47968300]
}, Open  ]],

Cell[CellGroupData[{

Cell["Adaptive Strategies", "Section",
 CellTags->"AdaptiveStrategies",
 CellID->155948475],

Cell["\<\
Adaptive strategies try to concentrate computational efforts where the \
integrand is discontinuous or has some other kind of singularity. Adaptive \
strategies differ by the way they partition the integration region into \
disjoint subregions. The integral estimates of each subregion contribute to \
the total integral estimate.\
\>", "Text",
 CellID->692243119],

Cell[TextData[{
 "The basic assumption for the adaptive strategies is that for given \
integration rule ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]], "InlineMath"],
 " and integrand ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 ", if an integration region ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 " is partitioned into, say, two disjoint subregions ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "1"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["V", "2"], " "}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"V", "=", 
    RowBox[{
     SubscriptBox["V", "1"], "\[Union]", 
     SubscriptBox["V", "2"]}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["V", "1"], "\[Intersection]", 
     SubscriptBox["V", "2"]}], "=", "0"}], TraditionalForm]], "InlineMath"],
 ", then the sum of the integral estimates of ",
 Cell[BoxData[
  FormBox["R", TraditionalForm]], "InlineMath"],
 " over ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "1"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "2"], TraditionalForm]], "InlineMath"],
 " is closer to the actual integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Integral]", "V"], 
    RowBox[{"f", 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ", that is,"
}], "Text",
 CellID->87866805],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"|", 
    FormBox[
     RowBox[{
      RowBox[{
       SubscriptBox["\[Integral]", "V"], 
       RowBox[{"f", 
        RowBox[{"\[DifferentialD]", "x"}]}]}], "-", 
      RowBox[{
       SubscriptBox["R", "V"], "(", "f", ")"}]}],
     TraditionalForm], "|", 
    RowBox[{">", 
     RowBox[{"|", 
      RowBox[{
       FormBox[
        RowBox[{
         SubscriptBox["\[Integral]", "V"], 
         RowBox[{"f", 
          RowBox[{"\[DifferentialD]", "x"}]}]}],
        TraditionalForm], "-", " ", 
       FormBox[
        RowBox[{
         RowBox[{
          SubscriptBox["R", 
           SubscriptBox["V", "1"]], "(", "f", ")"}], "+", 
         RowBox[{
          SubscriptBox["R", 
           SubscriptBox["V", "2"]], "(", "f", ")"}]}],
        TraditionalForm]}], "|"}]}]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"AdaptiveStrategies1",
 CellID->454180394],

Cell[TextData[{
 "and (",
 
 CounterBox["NumberedEquation", "AdaptiveStrategies1"],
 ") will imply that the sum of the error estimates for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["R", 
     SubscriptBox["V", "1"]], "(", "f", ")"}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["R", 
     SubscriptBox["V", "2"]], "(", "f", ")"}], TraditionalForm]], 
  "InlineMath"],
 " is smaller than the error estimate of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["R", "V"], "(", "f", ")"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->407570952],

Cell[TextData[{
 "Hence an adaptive strategy has these components [",
 ButtonBox["MalcSimp75",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#147637128"],
 "]:\n(i) an integration rule to compute the integral and error estimates \
over a region;\n(ii) a method for deciding which elements of a set of regions \
",
 Cell[BoxData[
  FormBox[
   SubsuperscriptBox[
    RowBox[{"{", 
     SubscriptBox["V", "i"], "}"}], 
    RowBox[{"i", "=", "1"}], "n"], TraditionalForm]], "InlineMath"],
 " to partition/subdivide;\n(iii) stopping criteria for deciding when to \
terminate the adaptive strategy algorithm. "
}], "Text",
 CellID->22798284]
}, Open  ]],

Cell[CellGroupData[{

Cell["Global Adaptive Strategy", "Section",
 CellTags->"GlobalAdaptive",
 CellID->137878679],

Cell["\<\
A global adaptive strategy reaches the required precision and accuracy goals \
of the integral estimate by recursive bisection of the subregion with the \
largest error estimate into two halves, and computes integral and error \
estimates for each half. \
\>", "Text",
 CellID->283998705],

Cell[TextData[{
 "The global adaptive algorithm for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " is specified with the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option value ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 StyleBox[".", "InlineCode"]
}], "MathCaption",
 CellID->70016952],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", "\"\<GlobalAdaptive\>\""}]}], "]"}]], "Input",
 CellLabel->"In[32]:=",
 CellID->29046796],

Cell[BoxData["2.000000000000003`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->7179300]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
integration rule used to compute integral and error estimate over each \
subregions\
\>", "TableText"]},
   {"\"\<SingularityDepth\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
number of recursive bisections before applying a singularity handler\
\>", "TableText"]},
   {"\"\<SingularityHandler\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "singularity handler", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->252682382],

Cell[TextData[{
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->343288445],

Cell[TextData[{
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " is the default integration strategy of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ". It is used for both one-dimensional and multidimensional integration. ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " works with both ",
 ButtonBox["Cartesian product rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#650745857"],
 " and ",
 ButtonBox["fully symmetric ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#89177434"],
 "multidimensional",
 ButtonBox[" rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#89177434"],
 "."
}], "Text",
 CellID->175887812],

Cell[TextData[{
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " uses a data structure called a \"heap\" to keep the set of regions \
partially sorted, with the largest error region being at the top of the heap. \
In the main loop of the algorithm the largest error region is bisected in the \
dimension that is estimated to be responsible for most of its error."
}], "Text",
 CellID->252624254],

Cell["\<\
It can be said that the algorithm produces the leaves of a binary tree, the \
nodes of which are the regions. The children of a node/region are its \
subregions obtained after bisection.\
\>", "Text",
 CellID->71598074],

Cell["\<\
After a bisection of a region and the subsequent integration over the new \
(sub)regions, new global integral and global error estimates are computed, \
which are sums of the integral and error estimates of all regions that are \
leaves of the binary tree.\
\>", "Text",
 CellID->276627368],

Cell[TextData[{
 "Each region has a record of how many bisections are made per dimension in \
order to produce it. When a region has been produced through too many \
bisections a singularity flattening algorithm is applied to it; see ",
 ButtonBox["Singularity Handling",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#122144792"],
 "."
}], "Text",
 CellID->395737582],

Cell[TextData[{
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " stops if the following expression is true:"
}], "Text",
 CellID->416354800],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"globalError", "\[LessEqual]", 
    RowBox[{"globalIntegral", " ", 
     SuperscriptBox["10", 
      RowBox[{"-", "pg"}]]}]}], "\[Or]", 
   RowBox[{"globalError", "\[LessEqual]", 
    SuperscriptBox["10", 
     RowBox[{"-", "ag"}]]}]}], ","}]], "NumberedEquation",
 CellTags->"GlobalAdaptiveStrategy1",
 CellID->199044157],

Cell[TextData[{
 "where ",
 Cell[BoxData["pg"], "InlineFormula"],
 " and ",
 Cell[BoxData["ag"], "InlineFormula"],
 " are precision and accuracy goals."
}], "Text",
 CellID->330012555],

Cell[TextData[{
 "The strategy also stops when the number of recursive bisections of a region \
exceeds a certain number (see ",
 ButtonBox["MinRecursion and MaxRecursion",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#280704764"],
 "), or when the global integration error oscillates too much (see ",
 ButtonBox["\"MaxErrorIncreases\"",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#285388386"],
 ")."
}], "Text",
 CellID->228203607],

Cell[TextData[{
 "Theoretical and practical evidence show that the global adaptive strategies \
have in general better performance than the local adaptive strategies [",
 ButtonBox["MalcSimp75",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#147637128"],
 "][",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "]."
}], "Text",
 CellID->8108703],

Cell[CellGroupData[{

Cell["MinRecursion and MaxRecursion", "Subsection",
 CellTags->"MinMaxRecursion",
 CellID->280704764],

Cell[TextData[{
 "The minimal and maximal depths of the recursive bisections are given by the \
values of the options ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 "."
}], "Text",
 CellID->487350830],

Cell[TextData[{
 "If for any subregion the number of bisections in any of the dimensions is \
greater than ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " then the integration by ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " stops. "
}], "Text",
 CellID->46226172],

Cell[TextData[{
 "Setting ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " to a positive integer forces recursive bisection of the integration \
regions before the integrand is ever evaluated. This can be done to ensure \
that a narrow spike in the integrand is not missed. (See ",
 ButtonBox["Tricking the error estimator",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#81420002"],
 ".)"
}], "Text",
 CellID->81043288],

Cell[TextData[{
 "For ",
 "multidimensional",
 " integration an effort is made to bisect in each dimension for each level \
of recursion in ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 "."
}], "Text",
 CellID->301375166]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"MaxErrorIncreases\"", "Subsection",
 CellTags->"MaxNumberOfErrorInreases",
 CellID->285388386],

Cell[TextData[{
 "Since (",
 
 CounterBox["NumberedEquation", "AdaptiveStrategies1"],
 ") is expected to hold in ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " the global error is expected to decrease after the bisection of the \
largest error region and the integration over its new parts. In other words \
the global error is expected to be more or less monotonically decreasing with \
respect to the number of integration steps."
}], "Text",
 CellID->67935078],

Cell[TextData[{
 "The global error might oscillate due to ",
 ButtonBox["phase errors",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#599865576"],
 " of the integration rules. Still, the global error is assumed at some point \
to start decreasing monotonically."
}], "Text",
 CellID->209598598],

Cell["\<\
Below are listed cases in which this assumption might become false.
(i) The actual integral is zero.\
\>", "Text",
 CellID->89602669],

Cell["Zero integral.", "MathCaption",
 CellID->121166150],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Sin", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"2", "\[Pi]"}]}], "}"}], ",", 
   RowBox[{"MaxRecursion", "->", "100"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->372749655],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->536853925]
}, Open  ]],

Cell["\<\
(ii) The specified working precision is not dense enough for the specified \
precision goal.\
\>", "Text",
 CellID->436824216],

Cell["The working precision is not dense enough.", "MathCaption",
 CellID->515392246],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"1", "/", 
     RowBox[{"Sqrt", "[", "x", "]"}]}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", " ", "0", ",", " ", "1"}], "}"}], ",", 
    RowBox[{"MaxRecursion", "\[Rule]", "100"}], ",", 
    RowBox[{"PrecisionGoal", "->", "17"}]}], "]"}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->704748196],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[33]:=",
 CellID->2518492],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"eincr\"\>"}], ":", 
  " ", "\<\"The global error of the strategy GlobalAdaptive has increased \
more than \\!\\(400\\) times. The global error is expected to decrease \
monotonically after a number of integrand evaluations. Suspect one of the \
following: the difference between the values of PrecisionGoal and \
WorkingPrecision is too small; the integrand is highly oscillatory or it is \
not a (piecewise) smooth function; or the true value of the integral is 0. \
Increasing the value of the GlobalAdaptive option MaxErrorIncreases might \
lead to a convergent numerical integration. NIntegrate obtained \
\\!\\(2.0000000000000018`\\) and \\!\\(2.1241892251243344`*^-16\\) for the \
integral and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/eincr\\\", ButtonNote -> \
\\\"NIntegrate::eincr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[33]:=",
 CellID->317353023],

Cell["2.0000000000000018", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]//InputForm=",
 CellID->68648145]
}, Open  ]],

Cell[TextData[{
 "(iii) The integration is badly conditioned [",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "]. For example, the reason might be that the integrand is defined by \
complicated expressions or in terms of approximate solutions of mathematical \
problems (such as differential equations or nonlinear algebraic equations)."
}], "Text",
 CellID->211340800],

Cell[TextData[{
 "The strategy ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " keeps track of the number of times the total error estimate has not \
decreased after the bisection of the region with the largest error estimate. \
When that number becomes bigger than the value of the ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " option ",
 Cell[BoxData["\"\<MaxErrorIncreases\>\""], "InlineFormula"],
 ", the integration stops with a message (",
 StyleBox["NIntegrate::eincr", "InlineCode"],
 ")."
}], "Text",
 CellID->717878841],

Cell[TextData[{
 "The default value of ",
 Cell[BoxData["\"\<MaxErrorIncreases\>\""], "InlineFormula"],
 " is 400 for one-dimensional integrals and 2000 for multidimensional \
integrals."
}], "Text",
 CellID->919099178],

Cell[TextData[{
 "The following integration invokes the message ",
 StyleBox["NIntegrate::eincr", "InlineCode"],
 ", with the default value of ",
 Cell[BoxData["\"\<MaxErrorIncreases\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->315283307],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Sin", "[", 
    RowBox[{
     SuperscriptBox["x", "2"], "+", "x"}], "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0", ",", 
     RowBox[{"80", "Pi"}]}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<GlobalAdaptive\>\""}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "100"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->353287776],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[1]:=",
 CellID->2036321957],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"eincr\"\>"}], ":", 
  " ", "\<\"The global error of the strategy GlobalAdaptive has increased \
more than \\!\\(400\\) times. The global error is expected to decrease \
monotonically after a number of integrand evaluations. Suspect one of the \
following: the difference between the values of PrecisionGoal and \
WorkingPrecision is too small; the integrand is highly oscillatory or it is \
not a (piecewise) smooth function; or the true value of the integral is 0. \
Increasing the value of the GlobalAdaptive option MaxErrorIncreases might \
lead to a convergent numerical integration. NIntegrate obtained \
\\!\\(2.972314689667426`\\) and \\!\\(9.140875003915308`\\) for the integral \
and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/eincr\\\", ButtonNote -> \
\\\"NIntegrate::eincr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[1]:=",
 CellID->1642367917],

Cell[BoxData["0``-0.9609877702615569"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->782217265]
}, Open  ]],

Cell[TextData[{
 "Increasing ",
 Cell[BoxData["\"\<MaxErrorIncreases\>\""], "InlineFormula"],
 " silences the ",
 StyleBox["NIntegrate::eincr", "InlineCode"],
 " message."
}], "MathCaption",
 CellID->5957940],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"res", "=", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", "x"}], "]"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"x", ",", " ", "0", ",", 
      RowBox[{"80", "Pi"}]}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"\"\<MaxErrorIncreases\>\"", "\[Rule]", "10000"}]}], "}"}]}], 
    ",", 
    RowBox[{"MaxRecursion", "\[Rule]", "20"}]}], "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->60671287],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[2]:=",
 CellID->931863114],

Cell[BoxData["0.5332463384612616`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1838525243]
}, Open  ]],

Cell["The result compares well with the exact value.", "MathCaption",
 CellID->38096366],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"exact", "=", 
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", "x"}], "]"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"x", ",", " ", "0", ",", 
       RowBox[{"80", "Pi"}]}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Abs", "[", 
  RowBox[{"res", "-", "exact"}], "]"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->8580501],

Cell[BoxData["6.840084054715589`*^-13"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->2006904466]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example implementation of a global adaptive strategy", "Subsection",
 CellTags->"GlobalAdaptiveImplementation",
 CellID->16672882],

Cell["\<\
This computes Gauss\[Dash]Kronrod abscissas, weights, and error weights.\
\>", "MathCaption",
 CellID->377226797],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
   RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
    RowBox[{"5", ",", "MachinePrecision"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[15]:=",
 CellID->194129872],

Cell[TextData[{
 "This is a definition of a function that applies the integration rule with \
abscissas and weights computed to the function ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " over the interval ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["a", "TI"], ",", 
    StyleBox["b", "TI"]}], "}"}]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->494545795],

Cell[BoxData[
 RowBox[{
  RowBox[{"IRuleEstimate", "[", 
   RowBox[{"f_", ",", 
    RowBox[{"{", 
     RowBox[{"a_", ",", "b_"}], "}"}]}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"integral", ",", "error"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"integral", ",", "error"}], "}"}], "=", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"b", "-", "a"}], ")"}], 
       RowBox[{"Total", "@", 
        RowBox[{"MapThread", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"Rescale", "[", 
             RowBox[{"absc", ",", 
              RowBox[{"{", 
               RowBox[{"0", ",", "1"}], "}"}], ",", 
              RowBox[{"{", 
               RowBox[{"a", ",", "b"}], "}"}]}], "]"}], ",", "weights", ",", 
            "errweights"}], "}"}]}], "]"}]}]}]}], ";", "\[IndentingNewLine]", 
     
     RowBox[{"{", 
      RowBox[{"integral", ",", 
       RowBox[{"Abs", "[", "error", "]"}]}], "}"}]}]}], "\[IndentingNewLine]",
    "]"}]}]], "Input",
 CellLabel->"In[16]:=",
 CellID->510565351],

Cell[TextData[{
 "This is a definition of a simple global adaptive algorithm that finds the \
integral of the function ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " over the interval ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["aArg", "TI"], ",", 
    StyleBox["bArg", "TI"]}], "}"}]], "InlineFormula"],
 " with relative error ",
 Cell[BoxData[
  StyleBox["tol", "TI"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->112552526],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"IStrategyGlobalAdaptive", "[", 
    RowBox[{"f_", ",", 
     RowBox[{"{", 
      RowBox[{"aArg_", ",", "bArg_"}], "}"}], ",", "tol_"}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "t", ",", "integral", ",", " ", "error", ",", "regions", ",", " ", "r1",
        ",", "r2", ",", 
       RowBox[{"a", "=", "aArg"}], ",", 
       RowBox[{"b", "=", "bArg"}], ",", "c"}], "}"}], ",", 
     "\[IndentingNewLine]", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"integral", ",", "error"}], "}"}], "=", 
       RowBox[{"IRuleEstimate", "[", 
        RowBox[{"f", ",", 
         RowBox[{"{", 
          RowBox[{"a", ",", "b"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{"boundaries", ",", " ", "integral", ",", " ", "error"}], " ", 
       "*)"}], "\[IndentingNewLine]", 
      RowBox[{"regions", "=", 
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"a", ",", "b"}], "}"}], ",", "integral", ",", "error"}], 
         "}"}], "}"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{"While", "[", 
       RowBox[{
        RowBox[{"error", ">=", " ", 
         RowBox[{"tol", "*", "integral"}]}], ",", "\[IndentingNewLine]", 
        RowBox[{"(*", " ", 
         RowBox[{
         "splitting", " ", "of", " ", "the", " ", "region", " ", "with", " ", 
          "the", " ", "largest", " ", "error"}], " ", "*)"}], 
        "\[IndentingNewLine]", 
        RowBox[{
         RowBox[{
          RowBox[{"{", 
           RowBox[{"a", ",", "b"}], "}"}], "=", 
          RowBox[{"regions", "\[LeftDoubleBracket]", 
           RowBox[{"1", ",", "1"}], "\[RightDoubleBracket]"}]}], ";", 
         RowBox[{"c", "=", 
          FractionBox[
           RowBox[{"a", "+", "b"}], "2"]}], ";", "\[IndentingNewLine]", 
         "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{
          "integration", " ", "of", " ", "the", " ", "left", " ", "region"}], 
          " ", "*)"}], "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"integral", ",", "error"}], "}"}], "=", 
          RowBox[{"IRuleEstimate", "[", 
           RowBox[{"f", ",", 
            RowBox[{"{", 
             RowBox[{"a", ",", "c"}], "}"}]}], "]"}]}], ";", 
         "\[IndentingNewLine]", 
         RowBox[{"r1", "=", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"a", ",", "c"}], "}"}], ",", "integral", ",", "error"}], 
           "}"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{
          "integration", " ", "of", " ", "the", " ", "right", " ", "region"}],
           " ", "*)"}], "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"integral", ",", "error"}], "}"}], "=", 
          RowBox[{"IRuleEstimate", "[", 
           RowBox[{"f", ",", 
            RowBox[{"{", 
             RowBox[{"c", ",", "b"}], "}"}]}], "]"}]}], ";", 
         "\[IndentingNewLine]", 
         RowBox[{"r2", "=", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"c", ",", "b"}], "}"}], ",", "integral", ",", "error"}], 
           "}"}]}], ";", "\[IndentingNewLine]", "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{"sort", " ", "the", " ", 
           RowBox[{"regions", ":", " ", 
            RowBox[{
            "the", " ", "largest", " ", "error", " ", "one", " ", "is", " ", 
             "the", " ", "first"}]}]}], " ", "*)"}], "\[IndentingNewLine]", 
         RowBox[{"regions", "=", 
          RowBox[{"Join", "[", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"r1", ",", "r2"}], "}"}], ",", 
            RowBox[{"Rest", "[", "regions", "]"}]}], "]"}]}], ";", 
         "\[IndentingNewLine]", 
         RowBox[{"regions", "=", 
          RowBox[{"Sort", "[", 
           RowBox[{"regions", ",", 
            RowBox[{
             RowBox[{
              RowBox[{
              "#1", "\[LeftDoubleBracket]", "3", "\[RightDoubleBracket]"}], 
              ">", 
              RowBox[{
              "#2", "\[LeftDoubleBracket]", "3", "\[RightDoubleBracket]"}]}], 
             "&"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
         "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{"global", " ", "integral", " ", "and", " ", "error"}], " ", 
          "*)"}], "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"integral", ",", "error"}], "}"}], "=", 
          RowBox[{"Total", "[", 
           RowBox[{"Map", "[", 
            RowBox[{
             RowBox[{
              RowBox[{"Rest", "[", "#1", "]"}], "&"}], ",", "regions"}], 
            "]"}], "]"}]}], ";"}]}], "\[IndentingNewLine]", 
       "\[IndentingNewLine]", "]"}], ";", "\[IndentingNewLine]", 
      "\[IndentingNewLine]", "integral"}]}], "\[IndentingNewLine]", "]"}]}], 
  ";"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->141774676],

Cell["This defines an integrand.", "MathCaption",
 CellID->739846109],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  RowBox[{"1", "/", 
   RowBox[{"Sqrt", "[", "x", "]"}]}]}]], "Input",
 CellLabel->"In[18]:=",
 CellID->273978834],

Cell["\<\
The global adaptive strategy defined earlier gives the following result.\
\>", "MathCaption",
 CellID->660429451],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"res", "=", 
  RowBox[{"IStrategyGlobalAdaptive", "[", 
   RowBox[{"f", ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}], ",", 
    SuperscriptBox["10", 
     RowBox[{"-", "8"}]]}], "]"}]}]], "Input",
 CellLabel->"In[19]:=",
 CellID->128801968],

Cell[BoxData["1.9999999947318126`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->43929232]
}, Open  ]],

Cell["Here is the exact result.", "MathCaption",
 CellID->12774559],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[20]:=",
 CellID->69911375],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->550003874]
}, Open  ]],

Cell["The relative error is within the prescribed tolerance.", "MathCaption",
 CellID->150740760],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"res", "-", "exact"}], "]"}], "/", "exact"}]], "Input",
 CellLabel->"In[21]:=",
 CellID->55456756],

Cell[BoxData["2.634093720921271`*^-9"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->855769821]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Local Adaptive Strategy", "Section",
 CellTags->"LocalAdaptive",
 CellID->19475523],

Cell["\<\
In order to reach the required precision and accuracy goals of the integral \
estimate, a local adaptive strategy recursively partitions the subregion into \
smaller disjoint subregions and computes integral and error estimates for \
each of them. \
\>", "Text",
 CellID->13896142],

Cell[TextData[{
 "The local adaptive algorithm for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " is specified with the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option value ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->625449359],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", "\"\<LocalAdaptive\>\""}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->863668896],

Cell[BoxData["1.9999999976742142`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->565424481]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
integration rule used to compute integral and error estimates over the \
subregions\
\>", "TableText"]},
   {"\"\<SingularityDepth\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
number of recursive bisections before applying a singularity handler\
\>", "TableText"]},
   {"\"\<SingularityHandler\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "singularity handler", "TableText"]},
   {"\"\<Partitioning\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
how to partition the regions in order to improve their integral estimate\
\>", "TableText"]},
   {"\"\<InitialEstimateRelaxation\>\"", 
    ButtonBox["True",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/True"], Cell["\<\
attempts to adjust the magnitude of the initial integral estimate in order to \
avoid unnecessary computation\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->468290175],

Cell[TextData[{
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->638537924],

Cell[TextData[{
 "Like ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " can be used for both one-dimensional and multidimensional integration. ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " works with both ",
 ButtonBox["Cartesian product rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#650745857"],
 " and ",
 ButtonBox["fully symmetric ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#89177434"],
 "multidimensional",
 ButtonBox[" rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#89177434"],
 "."
}], "Text",
 CellID->717575013],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " strategy has an initialization routine and a Recursive Routine (RR). RR \
produces the leaves of a tree, the nodes of which are regions. The children \
of a node/region are subregions obtained by its partition. RR takes a region \
as an argument and returns an integral estimate for it."
}], "Text",
 CellID->339839422],

Cell["\<\
RR uses an integration rule to compute integral and error estimates of the \
region argument. If the error estimate is too big, RR calls itself on the \
region's disjoint subregions obtained by partition. The sum of the integral \
estimates returned from these recursive calls becomes region's integral \
estimate. 
RR makes the decision to continue the recursion knowing only the integral and \
error estimates of the region at which it is executed. (This is why the \
strategy is called \"local adaptive\".)\
\>", "Text",
 CellID->704618760],

Cell["\<\
The initialization routine computes an initial estimation of the integral \
over the initial regions. This initial integral estimate is used in the \
stopping criteria of RR: if the error of a region is significant compared to \
the initial integral estimate then that region is partitioned into disjoint \
regions and RR is called on them; if the error is insignificant the recursion \
stops.\
\>", "Text",
 CellID->74249556],

Cell[TextData[{
 "The error estimate of a region, ",
 StyleBox["regionError", "InlineCode"],
 ", is considered insignificant if"
}], "Text",
 CellID->287350206],

Cell[BoxData[
 RowBox[{
  RowBox[{"initialIntegral", "+", "regionError"}], "==", 
  RowBox[{"initialIntegral", 
   StyleBox[".", "Text"]}]}]], "NumberedEquation",
 CellTags->"LocalAdaptiveStrategy1",
 CellID->174869493],

Cell[TextData[{
 "The stopping criteria (",
 
 CounterBox["NumberedEquation", "LocalAdaptiveStrategy1"],
 ") will compute the integral to the working precision. Since you want to \
compute the integral estimate to user-specified precision and accuracy goals, \
the following stopping criteria is used:"
}], "Text",
 CellID->13557248],

Cell[BoxData[{
 RowBox[{
  RowBox[{"integralEst", "=", 
   RowBox[{"Min", "[", 
    RowBox[{
     RowBox[{"initialIntegral", " ", 
      RowBox[{
       SuperscriptBox["10", 
        RowBox[{"-", "pg"}]], "/", "eps"}]}], ",", 
     RowBox[{
      SuperscriptBox["10", 
       RowBox[{"-", "ag"}]], "/", "eps"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"integralEst", "+", "regionError"}], "==", "integralEst"}], 
  ","}]}], "NumberedEquation",
 CellTags->"LocalAdaptiveStrategy2",
 CellID->108043378],

Cell[TextData[{
 "where ",
 Cell[BoxData["eps"], "InlineFormula"],
 " is the smallest number such that ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"1", "+", "eps"}], "\[NotEqual]", "1"}]], "InlineFormula"],
 " at the working precision, and ",
 Cell[BoxData["pg"], "InlineFormula"],
 " and ",
 Cell[BoxData["ag"], "InlineFormula"],
 " are the user-specified precision and accuracy goals."
}], "Text",
 CellID->76459917],

Cell[TextData[{
 "The recursive routine of ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " stops the recursion if:"
}], "Text",
 CellID->404870576],

Cell["\<\
there are no numbers of the specified working precision between region's \
boundaries;\
\>", "ItemizedText",
 CellID->12492713],

Cell["the maximum recursion level is reached;", "ItemizedText",
 CellID->32951193],

Cell[TextData[{
 "the error of the region is insignificant, i.e. the criteria (",
 
 CounterBox["NumberedEquation", "LocalAdaptiveStrategy2"],
 ") is true."
}], "ItemizedText",
 CellID->183321149],

Cell[CellGroupData[{

Cell["MinRecursion and MaxRecursion", "Subsection",
 CellTags->"GlobalAdaptiveImplementation",
 CellID->270286642],

Cell[TextData[{
 "The options ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " for ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " have the same meaning and functionality as they do for ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ". See ",
 ButtonBox["MinRecursion and MaxRecursion",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#280704764"],
 "."
}], "Text",
 CellID->238236236]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"InitialEstimateRelaxation\"", "Subsection",
 CellTags->"InitialEstimateRelaxation",
 CellID->64704670],

Cell[TextData[{
 " After the first recursion is finished a better integral estimate, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", "2"], TraditionalForm]], "InlineMath"],
 ", will be available. That better estimate is compared to the two integral \
estimates, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", "1"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", 
    RowBox[{"1", "e"}]], TraditionalForm]], "InlineMath"],
 " that the integration rule has used to give the integral estimate (",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", "1"], TraditionalForm]], "InlineMath"],
 ") and the error estimate (",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", 
    RowBox[{
     SubscriptBox["I", "1"], "-", 
     SubscriptBox["I", 
      RowBox[{"1", "e"}]]}], "|"}], TraditionalForm]], "InlineMath"],
 ") for the initial step. If"
}], "Text",
 CellID->592658558],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"\[Rho]", "=", 
    RowBox[{
     FractionBox[
      RowBox[{"|", 
       RowBox[{
        SubscriptBox["I", "2"], "-", 
        SubscriptBox["I", "1"]}], "|"}], 
      RowBox[{"|", 
       RowBox[{
        SubscriptBox["I", "2"], "-", 
        SubscriptBox["I", 
         RowBox[{"1", "e"}]]}], "|"}]], "<", "1"}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->7967936],

Cell[TextData[{
 "then the integral estimate ",
 StyleBox["integralEst", "InlineCode"],
 " in (",
 
 CounterBox["NumberedEquation", "LocalAdaptiveStrategy2"],
 ") can be increased\[LongDash]that is, the condition (",
 
 CounterBox["NumberedEquation", "LocalAdaptiveStrategy2"],
 ") is relaxed\[LongDash]with the formula"
}], "Text",
 CellID->578529350],

Cell[BoxData[
 RowBox[{
  RowBox[{"integralEst", " ", "=", " ", 
   RowBox[{"integralEst", " ", "/", "\[Rho]"}]}], ","}]], "DisplayMath",
 CellID->540469285],

Cell[TextData[{
 "since ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Rho]", "<", "1"}], TraditionalForm]], "InlineMath"],
 " means that the rule's integral estimate is more accurate than what the \
rule's error estimate predicts."
}], "Text",
 CellID->143101957]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"Partitioning\"", "Subsection",
 CellTags->"LAPartitioning",
 CellID->526467531],

Cell[TextData[{
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " has the option ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " to specify how are partition the regions that do not satisfy (",
 
 CounterBox["NumberedEquation", "LocalAdaptiveStrategy2"],
 "). For ",
 "one-dimensional",
 " integrals, if ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " is set to ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " partitions a region between the sampling points of the (rescaled) \
integration rule. In this way, if the integration rule is of ",
 ButtonBox["closed type",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#68931625"],
 ", every integration value can be reused. If ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " is given a list of integers, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["p", "1"], ",", 
     SubscriptBox["p", "2"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["p", "n"]}], "}"}], TraditionalForm]], "InlineMath"],
 " with length ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " that equals the number of integral variables, each dimension ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 " of the integration region is divided into ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["p", "i"], TraditionalForm]], "InlineMath"],
 " equal parts. If ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " is given an integer ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 StyleBox[",", "InlineCode"],
 " all dimensions are divided into ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 " equal parts."
}], "Text",
 CellID->151562619],

Cell["Consider the following function.", "MathCaption",
 CellID->152843209],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   FractionBox["1", 
    RowBox[{
     RowBox[{"10", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         FractionBox["1", "2"], "-", "x"}], ")"}], "2"]}], "+", "1"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->199405094],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzVWV2IVVUUPs6ZGcfy33EcbdTJnNHRmdF0HMffGXX8L83IfrC/UYuxHxUb
IyzCIiwkX+qh7KEQLLB6MBIjMIiESIiESIgKwiDsIQmSHgaC27f2Pvvuc769
Zu7tTj500Xv3Omevtb691rfW3ufM7T19vY882dO3Z1dP/foDPft79+x6qn7d
vgO4FA+LougM/h+sjTDOYWT+n8nl8Bt+N+G7DBNikWKZKVcjM5yYvldh79lP
xHfj/N2orMC9GgzkUplyz8yPZst3ub8TRRWErjqYUUnWxgczyoMrE4qYU0N2
42BG2QCYGZHurbxIHNcX6/DiojdoBorBLHmUPE+6jkidDx1hXCT20mMZ2gr9
DYyxgvz+l3l3tV6Z/NYkvxzTCanrDqWm52RbtWPJyokCV0XzLXP3g/e/vfDV
LzKsyOrs6T397smL0RtmPdeu9T976NPXX/uyv/9vU1vRKHyfzPqvzHs6/8XP
UL98+Q/uMxzh8gL3hyr/L+xX5eWPTl/at/fjq1f/yvl6s7Ednp+DDwKL8CLI
US3Zm0pyXQHZsLccST326nn8w0DNWJ4zdk+xbABzomNGBmDgAfhopEccRAns
efnI53ARHTXyjz/8Di18Gy6FWjZWn5z9HiGBx+gVI4OskK9c+TManUFm6+jS
d78ld4847YN9Z0V7jOpjuEMCLUQ1esnI4D2QynBsxscIcxfWE9wvyvXjb14Y
r9oe4WIF2zL7sJEx/Z23v5ahrnUDaT0vPqBRbSlhy8189JOB624jyc5ziZ2J
dt6guyWfHNjWocRWjWLL7atcGaPIxjO5pPuAFnkmKrbS+56sTcTRLg9Jrp92
uU6yNkmx5eLCFTjG2dq180PhSR/Zqi1gS37Hpfgs/D7gmIoqEXlykDvu+hzx
cc4C2CvD/SmeCsopKncMo2LUIUpZhnspTjdl2Fzt+Ai6y/AJIwNvwu461YfV
Aqxk13jcyFgo2lHOdZgBtIAcURYkj+WSOoP7nOtbeWR2T0U9YrbEr5eiMU31
YbUAA2Bk+ChldrqqZU9DICGoKMPdLgqIGbpJVB/kjk/raSbUkvZusl6vsIm5
wB24ltaxk9Z5c5E2RZxMug+T7Rkl4DNcjFEuKBoZPmRkkAgBkKFmc6BqnOLi
BzzSjx/M2U4zcwAbvMuITbvH2Z1A2HM/yTODjLqzFvesusT7DkLVoDJpKq37
PiOjFFGQMvRa4stwOMXxe4188ZtfgVKGjaoPq+W75t0O2QM7Tkldzcr4MGyL
YRBmZbidEM1SfUx3nIBNQXYXIZutatUTsjsJWVMQde351PHCcCZGaNCIZLgt
l62jJoUP4dmc82kqJdWn73AyHMlwTklWLVbfR7cS9uKtxvmV+7htobjOHcBa
rFTELblstd9mZJzW7tn+nqy/WbFV6IRrqifV7zeT3BzkOV1d8ttA/NpkZN/d
W1R+iZbU4kaKT2t+tthupLsbEq15qs1Gis86QqZrmapJ7YBrKQLzMohMtaTq
vJtWO1/1YbU839cQpwbTghfkF1mOVhN3bg1yEz6tuudKw9rUOajLyNgjEBux
vaAE7syhVXTSKjWb2tk0zuNDq0Uty3AlxXlhkbbS8lzK5AqSS7fpObWc8t+m
2NS7g2SlhSK2jCLaFmS4MokX56KZ8rmUuLJIZVgLzeogxrXntQRtq7ubnCg7
CH276sNq+apspwguVrVM1aX2ukWUu8UZZKZ+Up2/jdjUofqwWv6ks5Ci3xFE
f7A9zlRjqhsb9sefnfvphcPnZLikaGakPVir/mQ9nzK0dAhWvZX55EWzmj7j
iakFlNdWWv+ykpDZHPhTVwvlaChWPT+aSV4eZNr1Tbdqyz9/1ppLXFmuMqzN
xQQTJc7chVdkWGwrw1ej7f++pleqPtoJid3PfFUOpuUrpNFFJDlLdWaQddDq
GyhLnaqPxYTfnjX882yXqtVBvJrhYggr8kyzKshVetfLZs92Qd817IkRBYmy
lOEqhU2FdoAlhMeelP0OsFqx6WrGneyWEg+mEU/WlMTyZbS6qSQPxarHVkfY
u4u2Khmxe7DvivYJ0T+xrg2ym67ENC5ryfOcn9XXqeyy5wq/r/D7Aq8lPjsd
uuSNSw2tfb3qw2r5tzvVjiHJ+6MNqlYXzbJvfwATvBJ5YwbZ6tRs8WGt2adi
kTepPvJa9rV08uYNxZFobU6lz320vy26rtjtVorswWp0o8k70CKgdhnmGSWw
EZ5TmXXGcjkwYvWgCoJf5ZgDX/KuekuJlk2GR8IO9ikYRzItFeACSwBwXAd2
HD23/gsPMsHktQpGwA4EtHPFcesCaGF8W4l4TwxJT6ukU+ZS+PfraNg/wtMZ
6w==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->160223576]
}, Open  ]],

Cell[TextData[{
 "These are the sampling points used by ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " with its automatic region partitioning. It can be seen on the picture that \
the sampling points of each recursion level are between the sampling points \
of the previous recursion level."
}], "MathCaption",
 CellID->432814123],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sampledPoints", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        RowBox[{
         RowBox[{"10", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{
             FractionBox["1", "2"], "-", "x"}], ")"}], "2"]}], "+", "1"}]], 
       ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", "\"\<LocalAdaptive\>\"", "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"sampledPoints", ",", 
     RowBox[{"Range", "[", 
      RowBox[{"Length", "[", "sampledPoints", "]"}], "]"}]}], "}"}], "]"}], 
  "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->330513670],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxV1nlUlFUYBvDRVBJBOAHiQchhURZBRRBlk4dVVNZxhm1YZkUTQaLMUsov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     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageSize->{350.666666666667, Automatic},
  PlotRange->{{0., 1.}, {0., 177.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{355, 221},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->317842466]
}, Open  ]],

Cell[TextData[{
 "These are the sampling points used by ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " integration which partitions the regions with large error into three \
subregions. The patterns formed clearly show the three next recursion level \
subregions of each region of the first and second recursion levels. "
}], "MathCaption",
 CellID->453703361],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sampledPoints", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        RowBox[{
         RowBox[{"10", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{
             FractionBox["1", "2"], "-", "x"}], ")"}], "2"]}], "+", "1"}]], 
       ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<LocalAdaptive\>\"", ",", 
          RowBox[{"\"\<Partitioning\>\"", "\[Rule]", "3"}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"sampledPoints", ",", 
     RowBox[{"Range", "[", 
      RowBox[{"Length", "[", "sampledPoints", "]"}], "]"}]}], "}"}], "]"}], 
  "]"}]}], "Input",
 CellLabel->"In[5]:=",
 CellID->22061131],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlH1QVGUYxTemBHVMFKyGoVQKE0ckQWhFcw9fgnyzsAsLCziYK0ETajlL
BNMdB1GRwAaTKEs3mAmRGhqgLFjaJVRCofUj40MKA0FgWftDpYhxQjh3pt07
d+ed3549+z73Oc99V2fkyHfbSSSSA7Ofx6v19ZfM9FF2kS6/SsYvkHnkCZ+f
WtvIDog5V1P3tPE62RHz6yDZGarac2dzV46Qn8PIZzKfmNgJsiuqtleO+nZY
yKsg7ju/usG6npeQtb/M4eC/R6mvQfHOxErVoxryWjiXHdvSqvyBvA6KuesC
eT1WDw2XRLzfRd6A4DfWuRX7XiO/grZex5B3D90gb6T/V7K3TT0+OKVLr1fK
91HfBPuJYccPzx4n++J8o32BR5mO7AfXS/tm7zryqzCaJ1O9axvIUiSXSHsy
Qs+TN8OuK/sbnbqZ7E9/C3mLDW+Fn1F72FykJ7+Ge4KifkW0gbwNLaa8ya9O
ifnJbPoDWHq99DNLOuZZAJo9FNMn1J3UA7C006LZNHqZegD3v0I90JqFQHi0
37+qmhH1IBS8Xrrtn6Ju6kFw7+vVd2SZqAdj/9wfXKUejNa0Rve9L4rzFYLq
BaHBLeHMRwhB983y6cWOYj7brfMSbFgSirumyt0b3W9SD4UmQDiy6kAP9TC4
bJYOPznRTz2MeYvzvANvDw19ULDoT+o7EPnHj805F4aoh6PkWZ+oPZPD1MO5
/x3qETb1RaD/+jP1n38i1heJp6p3aSzpYn2RMEc8LPQ//Bv1KEhLh2Zv1itE
4ZDJZL4t66MejTdzdy3esEesPxonw+qNBetvUY9hPmTBhiWx+CLER+v4HRmx
6FoheO7MH+DvY9GgXx5ZV/X7PBtibfoThzW6h70eX5MRB5cHsrFfAm7THwev
VE27tppsiGPeZIncmiFHU//LK41jol+OzqDo6cxI9t8gR2tFxcKlg2RJPIbn
G0R/PBb1Feoztop5xOORZ/cVBxPZEA+nk/e/rPcW80mA8v95IYHPx/NISIBf
28DUso/v0Z+AqaNZGpe94vmlwPOlpdK1TWb6Faidu8bpV0B66aKr55m79CuQ
dnnc4n9HPB+VeKFcmxqkF/dXWs+PoLSux6CEHqflKQOsR5KImp5cbUMeGYlo
LPTKXDLK+oREzHVHSjYkIjMwsPH74kn6k/CO5e9qpymx/iSkF137OfsEWUji
vJANNixRodd+WY5bjOhXof30eHmemee9oIIpQld4I4lsUFn3R5KMwoWNJmFm
jP5kHG/KMvZlkoVkvBcVPOK9gGxI5vvAfkpSrBkpeNB98NiYgSykYFCjrg9I
E/ufAvNyJ92nnaP0q3Hx8eO4kqHGGZP7ZOhbzEdQoy7/1oCuj3kY1Khw/tZu
WivOTyoSrOYnFf8B740acA==
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 1.}, {0., 117.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->135681551]
}, Open  ]],

Cell[TextData[{
 "Multidimensional example of using the ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " option. To make the plot, the sampling points of the first region to be \
integrated, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"[", 
     RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
    RowBox[{"[", 
     RowBox[{"0", ",", "1"}], "]"}]}], TraditionalForm]], "InlineMath"],
 ", are removed."
}], "MathCaption",
 CellID->41412267],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"sampledPoints", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"x", "+", "y"}], ")"}], "6"], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<LocalAdaptive\>\"", ",", 
          RowBox[{"\"\<Partitioning\>\"", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"3", ",", "4"}], "}"}]}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sampledPoints", "=", 
   RowBox[{"Partition", "[", 
    RowBox[{"sampledPoints", ",", 
     RowBox[{
      RowBox[{"Length", "[", "sampledPoints", "]"}], "/", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"3", "*", "4"}], "+", "1"}], ")"}]}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"sampledPoints", "=", 
   RowBox[{"Flatten", "[", 
    RowBox[{
     RowBox[{"Rest", "[", "sampledPoints", "]"}], ",", "1"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"sampledPoints", ",", 
   RowBox[{"AspectRatio", "\[Rule]", "1"}], ",", 
   RowBox[{"GridLines", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Range", "[", "3", "]"}], "/", "3"}], ",", 
      RowBox[{
       RowBox[{"Range", "[", "4", "]"}], "/", "4"}]}], "}"}]}]}], 
  "]"}]}], "Input",
 CellLabel->"In[7]:=",
 CellID->485984991],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXL1uFDEQdnbv9oTYQ0CyR8pcByX0KEIIqEgqlFCeIsSlSIJCHuA6hARK
kYZIVAiQaCGCB6CBipaaFzn8s/aasfffm/UtPun2duzxzNietWc+r+7h5Gj6
ZG9ytLszWXtwOHk23d15vnb/4BAX+UsIoTP8/byK8P0c39Hv2XyOf+XrDXz1
cJVPKJ/wkFJEbyO5rs/q2AfBWl/UIi+nboRvSJGnqaP86Dq59pIahPrAuhWF
IwDSriocPaVkuQDPCMj1FQ4vxWZokV5br6Adzdo6KDZ6mTNQxGYyj2SerzVo
Kdeht9AvaHv1sVRlqfrSbewDvSbnnT/rQfw7in/hmC5L5dxKXTtOs6dWXlUu
A4k3KT2bzQrSfeukDPH1HZ+fROb67dN7d98qswZpOmfxCrr16GPcCq6OWikX
BM0a3ll/w26QZuZ6Whrz4yKiEa5JWo0B65dQhBvCtRrqAZbKvZyz9QJaCjV7
3FK5Z8k4H+x/+fb1d+zpP3/8iWk4gpQ/YPXTpy8FI9R+Eci5pB0HVc4VUj6Q
+5nIkG3T9CDFu9p8Etgsfnj/C4/561ff0RjQAZDSNH+WlclTMwa0Xmtz/FlW
Jt4wBrRea3P8iZVkhb5lkdeZkTICvrQKWtWtz9Ka+IZeavX6LK3JXOulVq83
PU8d87jYQ4bzfz0myKEpf8Dox9vHoiLMaTcAdGhSji/kwP0p9sgh8NAgh2a9
BKEJrghz2g0AHZqU4ws5KXHEEDwRQQ7NeqnGA2FOuwGgQ5NyfCHH0ljDtnXA
7Rl27hlxAkk/OhwKZhsVtWQiSk3ogNkW6RshTciGGY0+Q7Mttq/SB9si/zw0
Rp+D25YZyM8cx3FgTt72/iFHX3DtpbQmKorq8OVGR3AtZtLVqCWqw6dELyT6
IAPk0QVxc+NkRj9ohVz87a1jzI1L0YjQgaDFTVSHT6I9TOvxr3bGJQ8dYtLV
aCuqw2d91GX/LuAQHofwuGh9UaP1DuaHDuFxCA+1znaER8mUlbcyzMTw+Xms
aR38XQh+Rm9KppzjNzE2/CQfyj6f3auMLU3vabJvwsweesz/ud/Ztmu67KAK
/6JlBzb5rkN0HKLjEJ06UtyaXYV/0dZsazJjo1K6jOiknWGS2/SYvOkIuOxZ
lbGsqyUtJjNHv8GMUfsubadRKJgderJfOGRq4ZEp2/Yd23bQLu997jTDRr91
mUIV/kXLFGzyXYfuOHTHoTtlpJTNGu1c06u9EWnbmt+R9zpTTwSNnqouLopQ
4L1XF6t3PVa3JjvsNObj3jxy+E7nn+C41O0ZXd8zbPNbO3MBh+/Yhu8w5fyj
+9ctM6dcULL6L0TwpKsdjKicne2hJVo7+7KdfBoIE8N+9vdebG6ckF2bYUaC
FjdRHT6JTrClEuN5Xv5VS48uQ/xEi9R/z0NLfwEypj5R\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 178},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->19245743]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reuse of integrand values", "Subsection",
 CellID->214095646],

Cell[TextData[{
 "With its default partitioning settings for one-dimensional integrals ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " reuses the integrand values at the end points of the sub-intervals that \
have integral and error estimates that do not satisfy (",
 
 CounterBox["NumberedEquation", "LocalAdaptiveStrategy2"],
 ")."
}], "Text",
 CellID->91704383],

Cell[TextData[{
 "Sampling points of the integration of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SuperscriptBox["x", "6"], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " by ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 ". The variable ",
 StyleBox["rulePoints", "InlineCode"],
 " determines the number of points in the integration rule used by ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->163117692],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"rulePoints", "=", "5"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"sampledPoints", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       SuperscriptBox["x", "6"], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<LocalAdaptive\>\"", ",", 
          RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}], ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<ClenshawCurtisRule\>\"", ",", 
             RowBox[{"\"\<Points\>\"", "\[Rule]", "rulePoints"}]}], "}"}]}], 
          ",", 
          RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], 
         "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Length", "[", "sampledPoints", "]"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"sampledPoints", ",", 
     RowBox[{"Range", "[", 
      RowBox[{"Length", "[", "sampledPoints", "]"}], "]"}]}], "}"}], "]"}], 
  "]"}]}], "Input",
 CellLabel->"In[13]:=",
 CellID->130476304],

Cell[BoxData["65"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[15]=",
 CellID->483617285],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxV02tIU2Ecx/EVGeElB0qTpaWmOZdJ3jW1/eZl3qazMY0cJZlBeK1BERU5
UHSRvggt0FzepmgTX5gXwiWWmBpWsiZhiFnzvqZZKhLd3f+8aDAePufLs2dn
+x+3rELphZ0sFkv49/1v/f/1RTB+L7e04WaTgC7gompHwGD/c/IeSLSt7Xuf
vSWzYVlnyI44/Ujbdu3gPNkJ82pBgCTVRHZGk6h6IWhkhewK5lzL6o7f7cL8
l32nyB7w/JmtuGO4Sj4Mc7/cxs1eRebB8n0ryXzwxd5lnWs1ZB8c2TzgxG6s
J/vS/TSSj8GOl9Igq24m+2GE4znrrNaS/aHJ2WobTukkB2Aqv7bKpO4lB8Lx
MorszTpyEJa9KuWangFyMCaN8Xn3Q5jfLwQFzh6L7NuD5FC8tuGUHB0dIodB
GvTDcN5rlHwcxYXeokuur8jheC/TyvvLx8kRqF1yKZkM15MjIT7H0S/Imf/n
BLpLu3SZoQayAMVlZ3p55RNk4IHH+mqH6p3FSqA1TZZeNjNJXYiu7f1T1IUY
O5S0MRc4TT0KNdv7P1CPwmOzlao1nJmHaHALMqwq5sjKaLTomt3GjR+px8Bn
YkKmszdSj6H9s9RjIYg4qTGFzVGPpf1klggZkvV9tlKaN6UI/jZrkQ6fmPmL
Q1UaX1//ZIF6HFw1V1aMvYvU49H+dcDaq2eJejzsIr8/7KpYpp6A2a3NoT4X
ml9lAqyHzdlcBTPPidgdvH8pXc/0REyvcdhnsz5TT0KMIq8iytZMPYnOI7PE
KOl46ucroudBKUaRi6k7p4V5PpLBzxzluq+vUk+m88isFOy6q78xWMf0FEib
xr5xs5kuQV33i1wFj+kSqNm3fk1vMJ+fiusLb8YcDGSk4g/B/CnI
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 1.}, {0., 65.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 117},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->742613847]
}, Open  ]],

Cell["The percent of reused points in the integration.", "MathCaption",
 CellID->419411303],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"totalRulePoints", "=", 
   RowBox[{
    RowBox[{"2", "rulePoints"}], "-", "1"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"totalPoints", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"(", 
      RowBox[{"totalRulePoints", "-", "1"}], ")"}], "totalRulePoints"}], "+", 
    "totalRulePoints"}]}], ";"}], "\n", 
 RowBox[{
  FractionBox[
   RowBox[{"totalPoints", "-", 
    RowBox[{"Length", "[", "sampledPoints", "]"}]}], "totalPoints"], "//", 
  "N"}]}], "Input",
 CellLabel->"In[17]:=",
 CellID->479980774],

Cell[BoxData["0.19753086419753085`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->2017206282]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example implementation of a local adaptive strategy", "Subsection",
 CellTags->"GlobalAdaptiveImplementation",
 CellID->715598439],

Cell["\<\
This computes Clenshaw\[Dash]Curtis abscissas, weights, and error weights.\
\>", "MathCaption",
 CellID->584270],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"absc", ",", "weights", ",", "errweights"}], "}"}], "=", 
   RowBox[{"NIntegrate`ClenshawCurtisRuleData", "[", 
    RowBox[{"6", ",", "MachinePrecision"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->478557852],

Cell[TextData[{
 "This is a definition of a function that applies the integration rule with \
computed abscissas and weights computed above to the function ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " over the interval ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["a", "TI"], ",", 
    StyleBox["b", "TI"]}], "}"}]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->161574657],

Cell[BoxData[
 RowBox[{
  RowBox[{"IRuleEstimate", "[", 
   RowBox[{"f_", ",", 
    RowBox[{"{", 
     RowBox[{"a_", ",", "b_"}], "}"}]}], "]"}], ":=", "\[IndentingNewLine]", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"integral", ",", "error", ",", "scaledAbsc"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"scaledAbsc", "=", 
      RowBox[{"Rescale", "[", 
       RowBox[{"absc", ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"a", ",", "b"}], "}"}]}], "]"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"integral", ",", "error"}], "}"}], "=", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"b", "-", "a"}], ")"}], 
       RowBox[{"Total", "@", 
        RowBox[{"MapThread", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"{", 
            RowBox[{
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#2"}], ",", 
             RowBox[{
              RowBox[{"f", "[", "#1", "]"}], "#3"}]}], "}"}], "&"}], ",", 
          RowBox[{"{", 
           RowBox[{"scaledAbsc", ",", "weights", ",", "errweights"}], "}"}]}],
          "]"}]}]}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"{", 
      RowBox[{"integral", ",", 
       RowBox[{"Abs", "[", "error", "]"}], ",", "scaledAbsc"}], "}"}]}]}], 
   "\[IndentingNewLine]", "]"}]}]], "Input",
 CellLabel->"In[34]:=",
 CellID->570937231],

Cell[TextData[{
 "This is definition of a simple local adaptive algorithm, that finds the \
integral of the function ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " over the interval ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["aArg", "TI"], ",", 
    StyleBox["bArg", "TI"]}], "}"}]], "InlineFormula"],
 " with relative error ",
 Cell[BoxData[
  StyleBox["tol", "TI"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->214119439],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"LocalAdaptiveRecurence", "[", 
     RowBox[{"f_", ",", 
      RowBox[{"{", 
       RowBox[{"a_", ",", "b_"}], "}"}], ",", "integralEst_"}], "]"}], ":=", 
    "\[IndentingNewLine]", 
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"regions", ",", "integral", ",", "error", ",", "scaledAbsc"}], 
       "}"}], ",", "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{"integral", ",", "error", ",", "scaledAbsc"}], "}"}], "=", 
        RowBox[{"IRuleEstimate", "[", 
         RowBox[{"f", ",", 
          RowBox[{"{", 
           RowBox[{"a", ",", "b"}], "}"}]}], "]"}]}], ";", 
       "\[IndentingNewLine]", "\[IndentingNewLine]", 
       RowBox[{"If", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"N", "[", 
           RowBox[{"integralEst", "+", "error"}], "]"}], "==", 
          RowBox[{"N", "[", "integralEst", "]"}]}], ",", 
         "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{"Stopping", " ", "criteria", " ", "is", " ", "satisfied"}], 
          " ", "*)"}], "\[IndentingNewLine]", "integral", ",", 
         "\[IndentingNewLine]", 
         RowBox[{"(*", " ", 
          RowBox[{"ELSE", " ", "call", " ", "itself", " ", "recursively"}], 
          " ", "*)"}], "\[IndentingNewLine]", 
         RowBox[{
          RowBox[{"regions", " ", "=", " ", 
           RowBox[{"Partition", "[", 
            RowBox[{"scaledAbsc", ",", "2", ",", "1"}], "]"}]}], ";", 
          "\[IndentingNewLine]", 
          RowBox[{"Total", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"LocalAdaptiveRecurence", "[", 
              RowBox[{"f", ",", "#1", ",", "integralEst"}], "]"}], "&"}], "/@",
             "regions"}], "]"}]}]}], "\[IndentingNewLine]", "]"}]}]}], 
     "\[IndentingNewLine]", "]"}]}], ";"}], "\n"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"IStrategyLocalAdaptive", "[", 
    RowBox[{"f_", ",", 
     RowBox[{"{", 
      RowBox[{"aArg_", ",", "bArg_"}], "}"}], ",", "tol_"}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"integral", ",", " ", "error", ",", 
       RowBox[{"a", "=", "aArg"}], ",", 
       RowBox[{"b", "=", "bArg"}], ",", 
       RowBox[{"d", "=", "1"}], ",", "dummy"}], "}"}], ",", 
     "\[IndentingNewLine]", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"a", ">", "b"}], ",", 
        RowBox[{
         RowBox[{
          RowBox[{"{", 
           RowBox[{"a", ",", "b"}], "}"}], "=", 
          RowBox[{"{", 
           RowBox[{"b", ",", "a"}], "}"}]}], ";", 
         RowBox[{"d", "=", 
          RowBox[{"-", "1"}]}]}]}], "]"}], ";", "\[IndentingNewLine]", 
      "\[IndentingNewLine]", 
      RowBox[{"(*", " ", 
       RowBox[{"initial", " ", "integral", " ", "estimate"}], " ", "*)"}], 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"integral", ",", "error", ",", "dummy"}], "}"}], "=", 
       RowBox[{"IRuleEstimate", "[", 
        RowBox[{"f", ",", 
         RowBox[{"{", 
          RowBox[{"a", ",", "b"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", "\[IndentingNewLine]", 
      RowBox[{"d", "*", 
       RowBox[{"LocalAdaptiveRecurence", "[", 
        RowBox[{"f", ",", 
         RowBox[{"{", 
          RowBox[{"a", ",", "b"}], "}"}], ",", 
         RowBox[{"d", "*", "integral", "*", 
          RowBox[{"tol", "/", "$MachineEpsilon"}]}]}], "]"}]}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[35]:=",
 CellID->72899967],

Cell["This defines a function.", "MathCaption",
 CellID->610701939],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  RowBox[{
   RowBox[{"Sqrt", "[", "x", "]"}], "*", 
   RowBox[{"Sin", "[", "x", "]"}]}]}]], "Input",
 CellLabel->"In[37]:=",
 CellID->290109502],

Cell["The local adaptive strategy gives the result.", "MathCaption",
 CellID->207964102],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"res", "=", 
  RowBox[{"IStrategyLocalAdaptive", "[", 
   RowBox[{"f", ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", 
      RowBox[{"8", " ", "\[Pi]"}]}], "}"}], ",", 
    SuperscriptBox["10", 
     RowBox[{"-", "8"}]]}], "]"}]}]], "Input",
 CellLabel->"In[38]:=",
 CellID->303377663],

Cell[BoxData[
 RowBox[{"-", "4.388572155738283`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->161645810]
}, Open  ]],

Cell["This is the exact result.", "MathCaption",
 CellID->229013455],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", 
      RowBox[{"8", " ", "\[Pi]"}]}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[39]:=",
 CellID->475855474],

Cell[BoxData[
 RowBox[{
  SqrtBox[
   FractionBox["\[Pi]", "2"]], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", "4"}], "+", 
    RowBox[{"FresnelC", "[", "4", "]"}]}], ")"}]}]], "Output",
 ImageSize->{158, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->305941435]
}, Open  ]],

Cell["The relative error is within the prescribed tolerance.", "MathCaption",
 CellID->392660834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"res", "-", "exact"}], "]"}], "/", "exact"}]], "Input",
 CellLabel->"In[40]:=",
 CellID->1002516567],

Cell[BoxData[
 RowBox[{"-", "2.0305628811558452`*^-11"}]], "Output",
 ImageSize->{102, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[40]=",
 CellID->267646052]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"GlobalAdaptive\" versus \"LocalAdaptive\"", "Section",
 CellID->290792217],

Cell["\<\
In general the global adaptive strategy has better performance than the local \
adaptive one. In some cases though the local adaptive strategy is more robust \
and/or gives better performance. \
\>", "Text",
 CellID->166481573],

Cell[TextData[{
 "There are two main differences between ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 ":"
}], "Text",
 CellID->613752572],

Cell[TextData[{
 "1. ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " stops when the sum of the errors of all regions satisfies the precision \
goal, while ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " stops when the error of each region is small enough compared to an \
estimate of the integral. "
}], "Text",
 CellID->602896345],

Cell[TextData[{
 "2. To improve the integral estimate ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " bisects the region with largest error, while ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " partitions all regions the error for which is not small enough. "
}], "Text",
 CellID->174992263],

Cell[TextData[{
 "For ",
 "multidimensional",
 " integrals ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " is much faster because ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " does partitioning along each axis, so the number of regions can explode \
combinatorically."
}], "Text",
 CellID->810374672],

Cell[TextData[{
 "Why and how global adaptive strategy is faster for one-dimensional smooth \
integrands is proved (and explained) in [",
 ButtonBox["MalcSimp75",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#147637128"],
 "]."
}], "Text",
 CellID->374781156],

Cell[TextData[{
 "When ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " is faster and with better performance than ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ", it is because the precision goal stopping criteria and partitioning \
strategy of ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " are more suited for integrand's nature. Another factor is the ability of \
",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " to reuse the integral values of all points already sampled. ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " has the ability to reuse very few integral values. (At most 3 per rule \
application, 0 for the default ",
 "one-dimensional",
 " rule, the Gauss\[Dash]Kronrod rule.)"
}], "Text",
 CellID->556354165],

Cell[TextData[{
 "The following subsection demonstrates the performance differences between \
",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 "."
}], "Text",
 CellID->578117949],

Cell[CellGroupData[{

Cell["\"GlobalAdaptive\" is generally better than \"LocalAdaptive\"", \
"Subsubsection",
 CellID->558182960],

Cell[TextData[{
 "The table that follows, with timing ratios and numbers of integrand \
evaluations, demonstrates that ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " is better than ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " for the most common cases. All integrals considered are one-dimensional \
over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 ", because (i) for multidimensional integrals the performance differences \
should be expected to deepen, since ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " partitions the regions along each axis, and (ii) ",
 "one-dimensional",
 " integrals over different ranges can be rescaled to be over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->448689488],

Cell[TextData[{
 "Definitions of functions, precision goals, number of integrations, \
integration rule. The variable ",
 Cell[BoxData["integrationRule"], "InlineFormula"],
 " can be changed in order to make the profiling runs below with the same \
integration rule. The last function is derived from ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", "x"}]], 
    RowBox[{"sin", "(", "x", ")"}]}], TraditionalForm]], "InlineMath"],
 " by the variable change ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "\[Rule]", 
    RowBox[{
     RowBox[{"-", "1"}], "+", 
     FractionBox["1", 
      RowBox[{"1", "-", "x"}]]}]}], TraditionalForm]], "InlineMath"],
 " that transforms ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], ")"}], TraditionalForm]], "InlineMath"],
 " into ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "\[Infinity]"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->710209410],

Cell[BoxData[{
 RowBox[{
  RowBox[{"funcs", "=", 
   RowBox[{"{", 
    RowBox[{
     SqrtBox["x"], ",", 
     FractionBox["1", 
      SqrtBox["x"]], ",", 
     FractionBox[
      RowBox[{"Sin", "[", 
       RowBox[{"200", " ", "x"}], "]"}], 
      SqrtBox["x"]], ",", 
     RowBox[{"Log", "[", "x", "]"}], ",", 
     SuperscriptBox["x", "26"], ",", 
     FractionBox["1", 
      RowBox[{
       RowBox[{
        SuperscriptBox["10", "4"], 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           FractionBox["1", "2"], "-", "x"}], ")"}], "2"]}], "+", "1"}]], ",",
      
     RowBox[{"-", 
      FractionBox[
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"1", "-", 
          FractionBox["1", 
           RowBox[{"1", "-", "x"}]]}]], " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"1", "-", 
          FractionBox["1", 
           RowBox[{"1", "-", "x"}]]}], "]"}]}], 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"1", "-", "x"}], ")"}], "2"]]}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"precs", "=", 
   RowBox[{"{", 
    RowBox[{"6", ",", "8", ",", "10", ",", "12", ",", "14"}], "}"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"n", "=", "10"}], ";"}], " ", 
  RowBox[{"(*", " ", 
   RowBox[{
   "number", " ", "of", " ", "integrations", " ", "to", " ", "determine", " ",
     "the", " ", "timing"}], " ", "*)"}]}], "\n", 
 RowBox[{
  RowBox[{"integrationRule", "=", "Automatic"}], ";"}]}], "Input",
 CellLabel->"In[21]:=",
 CellID->781774074],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FRangesToCube", "[", 
  RowBox[{"{", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], "}"}], "]"}]], "Input",\

 CellLabel->"In[25]:=",
 CellID->533563938],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      FractionBox["1", 
       RowBox[{"1", "-", "x"}]]}]}], "}"}], ",", 
   FractionBox["1", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", "x"}], ")"}], "2"]]}], "}"}]], "Output",
 ImageSize->{175, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->1400684836]
}, Open  ]],

Cell[TextData[{
 "Exact integral values. All integrals are over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->1807556],

Cell[BoxData[
 RowBox[{
  RowBox[{"exactvals", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Integrate", "[", 
      RowBox[{"#", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], "&"}], "/@", 
    "funcs"}]}], ";"}]], "Input",
 CellLabel->"In[26]:=",
 CellID->157089714],

Cell[TextData[{
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " timings."
}], "MathCaption",
 CellID->2347770],

Cell[BoxData[
 RowBox[{
  RowBox[{"gatimings", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"First", "@", 
       RowBox[{"Timing", "[", 
        RowBox[{"Do", "[", 
         RowBox[{
          RowBox[{"NIntegrate", "[", 
           RowBox[{
            RowBox[{"#", "[", 
             RowBox[{"[", "1", "]"}], "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"PrecisionGoal", "\[Rule]", 
             RowBox[{"#", "[", 
              RowBox[{"[", "2", "]"}], "]"}]}], ",", 
            RowBox[{"Method", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
               RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}], ",", 
               RowBox[{"Method", "\[Rule]", "integrationRule"}], ",", 
               RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], 
              "}"}]}], ",", 
            RowBox[{"MaxRecursion", "\[Rule]", "200"}]}], "]"}], ",", 
          RowBox[{"{", "n", "}"}]}], "]"}], "]"}]}], "&"}], ",", 
     RowBox[{"Outer", "[", 
      RowBox[{"List", ",", "funcs", ",", "precs", ",", "1"}], "]"}], ",", 
     RowBox[{"{", "2", "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[32]:=",
 CellID->36839626],

Cell[TextData[{
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " timings."
}], "MathCaption",
 CellID->16771603],

Cell[BoxData[
 RowBox[{
  RowBox[{"latimings", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"First", "@", 
       RowBox[{"Timing", "[", 
        RowBox[{"Do", "[", 
         RowBox[{
          RowBox[{"NIntegrate", "[", 
           RowBox[{
            RowBox[{"#", "[", 
             RowBox[{"[", "1", "]"}], "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"PrecisionGoal", "\[Rule]", 
             RowBox[{"#", "[", 
              RowBox[{"[", "2", "]"}], "]"}]}], ",", 
            RowBox[{"Method", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{"\"\<LocalAdaptive\>\"", ",", 
               RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}], ",", 
               RowBox[{"Method", "\[Rule]", "integrationRule"}], ",", 
               RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], 
              "}"}]}], ",", 
            RowBox[{"MaxRecursion", "\[Rule]", "200"}]}], "]"}], ",", 
          RowBox[{"{", "n", "}"}]}], "]"}], "]"}]}], "&"}], ",", 
     RowBox[{"Outer", "[", 
      RowBox[{"List", ",", "funcs", ",", "precs", ",", "1"}], "]"}], ",", 
     RowBox[{"{", "2", "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[28]:=",
 CellID->248031282],

Cell[TextData[{
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " function evaluations."
}], "MathCaption",
 CellID->288666510],

Cell[BoxData[
 RowBox[{
  RowBox[{"ganfevals", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"k", "=", "0"}], ";", 
        RowBox[{"res", "=", 
         RowBox[{"NIntegrate", "[", 
          RowBox[{
           RowBox[{"#", "[", 
            RowBox[{"[", 
             RowBox[{"1", ",", "1"}], "]"}], "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"PrecisionGoal", "\[Rule]", 
            RowBox[{"#", "[", 
             RowBox[{"[", "2", "]"}], "]"}]}], ",", 
           RowBox[{"Method", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
              RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}], ",", 
              RowBox[{"Method", "\[Rule]", "integrationRule"}], ",", 
              RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], 
             "}"}]}], ",", 
           RowBox[{"MaxRecursion", "\[Rule]", "200"}], ",", 
           RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
            RowBox[{"k", "++"}]}]}], "]"}]}], ";", 
        RowBox[{"{", 
         RowBox[{"k", ",", 
          RowBox[{
           RowBox[{"Abs", "[", 
            RowBox[{"res", "-", 
             RowBox[{"#", "[", 
              RowBox[{"[", 
               RowBox[{"1", ",", "2"}], "]"}], "]"}]}], "]"}], "/", 
           RowBox[{"Abs", "[", 
            RowBox[{"#", "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "2"}], "]"}], "]"}], "]"}]}]}], "}"}]}], 
       ")"}], "&"}], ",", 
     RowBox[{"Outer", "[", 
      RowBox[{"List", ",", 
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"funcs", ",", "exactvals"}], "}"}], "]"}], ",", "precs", ",",
        "1"}], "]"}], ",", 
     RowBox[{"{", "2", "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[29]:=",
 CellID->306406282],

Cell[TextData[{
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " function evaluations."
}], "MathCaption",
 CellID->614791391],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"lanfevals", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"k", "=", "0"}], ";", 
        RowBox[{"res", "=", 
         RowBox[{"NIntegrate", "[", 
          RowBox[{
           RowBox[{"#", "[", 
            RowBox[{"[", 
             RowBox[{"1", ",", "1"}], "]"}], "]"}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"PrecisionGoal", "\[Rule]", 
            RowBox[{"#", "[", 
             RowBox[{"[", "2", "]"}], "]"}]}], ",", 
           RowBox[{"Method", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"\"\<LocalAdaptive\>\"", ",", 
              RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}], ",", 
              RowBox[{"Method", "\[Rule]", "integrationRule"}], ",", 
              RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], 
             "}"}]}], ",", 
           RowBox[{"MaxRecursion", "\[Rule]", "200"}], ",", 
           RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
            RowBox[{"k", "++"}]}]}], "]"}]}], ";", 
        RowBox[{"{", 
         RowBox[{"k", ",", 
          RowBox[{
           RowBox[{"Abs", "[", 
            RowBox[{"res", "-", 
             RowBox[{"#", "[", 
              RowBox[{"[", 
               RowBox[{"1", ",", "2"}], "]"}], "]"}]}], "]"}], "/", 
           RowBox[{"Abs", "[", 
            RowBox[{"#", "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "2"}], "]"}], "]"}], "]"}]}]}], "}"}]}], 
       ")"}], "&"}], ",", 
     RowBox[{"Outer", "[", 
      RowBox[{"List", ",", 
       RowBox[{"Transpose", "[", 
        RowBox[{"{", 
         RowBox[{"funcs", ",", "exactvals"}], "}"}], "]"}], ",", "precs", ",",
        "1"}], "]"}], ",", 
     RowBox[{"{", "2", "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[30]:=",
 CellID->121123080],

Cell[BoxData["$Aborted"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->1209214834]
}, Open  ]],

Cell["Table with the timing ratios and integrand evaluations.", "MathCaption",
 CellID->8143956],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Grid", "[", 
  RowBox[{
   RowBox[{"Prepend", "[", 
    RowBox[{
     RowBox[{"Transpose", "@", 
      RowBox[{"{", 
       RowBox[{"funcs", ",", 
        RowBox[{"ColumnForm", "/@", 
         RowBox[{"Table", "[", 
          RowBox[{"precs", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "funcs", "]"}], "}"}]}], "]"}]}], ",", 
        RowBox[{"ColumnForm", "/@", 
         RowBox[{"(", 
          RowBox[{"latimings", "/", "gatimings"}], ")"}]}], ",", 
        RowBox[{
         RowBox[{
          RowBox[{"ColumnForm", "[", 
           RowBox[{"First", "/@", "#"}], "]"}], "&"}], "/@", "ganfevals"}], 
        ",", 
        RowBox[{
         RowBox[{
          RowBox[{"ColumnForm", "[", 
           RowBox[{"First", "/@", "#"}], "]"}], "&"}], "/@", "lanfevals"}]}], 
       "}"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", "\"\<SmallText\>\""}], "]"}], "&"}], "/@", 
      RowBox[{"{", 
       RowBox[{
       "\"\<functions\>\"", ",", "\"\<precision goals\>\"", ",", 
        "\"\<\!\(\*FractionBox[\(LocalAdaptive\\\ timings\), \(GlobalAdaptive\
\\\ timings\)]\)\>\"", ",", "\"\<GlobalAdaptive\\nfunction\\nevaluations\>\"",
         ",", "\"\<LocalAdaptive\\nfunction\\nevaluations\>\""}], "}"}]}]}], 
    "]"}], ",", 
   RowBox[{"Frame", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[70]:=",
 CellID->519266640],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{553, 627},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[70]=",
 CellID->93033559]
}, Open  ]],

Cell[TextData[{
 "Table with the errors of the integrations. Both ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " reach the required precision goals."
}], "MathCaption",
 CellID->364839605],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Grid", "[", 
  RowBox[{
   RowBox[{"Prepend", "[", 
    RowBox[{
     RowBox[{"Transpose", "@", 
      RowBox[{"{", 
       RowBox[{"funcs", ",", 
        RowBox[{"ColumnForm", "/@", 
         RowBox[{"Table", "[", 
          RowBox[{"precs", ",", 
           RowBox[{"{", 
            RowBox[{"Length", "[", "funcs", "]"}], "}"}]}], "]"}]}], ",", 
        RowBox[{
         RowBox[{
          RowBox[{"ColumnForm", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"#", "[", 
              RowBox[{"[", "2", "]"}], "]"}], "&"}], "/@", "#"}], "]"}], 
          "&"}], "/@", "ganfevals"}], ",", 
        RowBox[{
         RowBox[{
          RowBox[{"ColumnForm", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"#", "[", 
              RowBox[{"[", "2", "]"}], "]"}], "&"}], "/@", "#"}], "]"}], 
          "&"}], "/@", "lanfevals"}]}], "}"}]}], ",", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", "\"\<SmallText\>\""}], "]"}], "&"}], "/@", 
      RowBox[{"{", 
       RowBox[{
       "\"\<functions\>\"", ",", "\"\<precision goals\>\"", ",", 
        "\"\<GlobalAdaptive\\nrelative errors\>\"", ",", 
        "\"\<LocalAdaptive\\nrelative errors\>\""}], "}"}]}]}], "]"}], ",", 
   RowBox[{"Frame", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[73]:=",
 CellID->213045600],

Cell[BoxData[
 TagBox[GridBox[{
    {
     StyleBox["\<\"functions\"\>", "SmallText",
      StripOnInput->False], 
     StyleBox["\<\"precision goals\"\>", "SmallText",
      StripOnInput->False], 
     StyleBox["\<\"GlobalAdaptive\\nrelative errors\"\>", "SmallText",
      StripOnInput->False], 
     StyleBox["\<\"LocalAdaptive\\nrelative errors\"\>", "SmallText",
      StripOnInput->False]},
    {
     SqrtBox["x"], 
     InterpretationBox[GridBox[{
        {"6"},
        {"8"},
        {"10"},
        {"12"},
        {"14"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{6, 8, 10, 12, 14}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"3.5974713874509234`*^-8"},
        {"5.621058618565655`*^-10"},
        {"3.1063485117499567`*^-12"},
        {"1.815214645262131`*^-14"},
        {"9.992007221626409`*^-16"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{3.5974713874509234`*^-8, 5.621058618565655*^-10, 
       3.1063485117499567`*^-12, 1.815214645262131*^-14, 
       9.992007221626409*^-16}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"1.8214283248330077`*^-8"},
        {"1.3520368158381757`*^-10"},
        {"1.0040301923197603`*^-12"},
        {"7.494005416219807`*^-15"},
        {"1.6653345369377348`*^-16"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{1.8214283248330077`*^-8, 1.3520368158381757`*^-10, 
       1.0040301923197603`*^-12, 7.494005416219807*^-15, 
       1.6653345369377348`*^-16}],
      Editable->False]},
    {
     FractionBox["1", 
      SqrtBox["x"]], 
     InterpretationBox[GridBox[{
        {"6"},
        {"8"},
        {"10"},
        {"12"},
        {"14"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{6, 8, 10, 12, 14}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"4.7682213921707017`*^-7"},
        {"2.6340936098989687`*^-9"},
        {"1.0288436769201326`*^-11"},
        {"5.562217353372034`*^-14"},
        {"1.1102230246251565`*^-15"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{4.7682213921707017`*^-7, 2.6340936098989687`*^-9, 
       1.0288436769201326`*^-11, 5.562217353372034*^-14, 
       1.1102230246251565`*^-15}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"2.164678349902971`*^-7"},
        {"1.607351474142149`*^-9"},
        {"1.1934897514720433`*^-11"},
        {"4.5208281562736374`*^-13"},
        {"3.219646771412954`*^-15"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{2.164678349902971*^-7, 1.607351474142149*^-9, 
       1.1934897514720433`*^-11, 4.5208281562736374`*^-13, 
       3.219646771412954*^-15}],
      Editable->False]},
    {
     FractionBox[
      RowBox[{"Sin", "[", 
       RowBox[{"200", " ", "x"}], "]"}], 
      SqrtBox["x"]], 
     InterpretationBox[GridBox[{
        {"6"},
        {"8"},
        {"10"},
        {"12"},
        {"14"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{6, 8, 10, 12, 14}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"1.3585647313216017`*^-8"},
        {"7.504991036667412`*^-11"},
        {"1.4618753322522816`*^-13"},
        {"6.117980465372985`*^-15"},
        {"8.0499742965434`*^-16"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{1.3585647313216017`*^-8, 7.504991036667412*^-11, 
       1.4618753322522816`*^-13, 6.117980465372985*^-15, 
       8.0499742965434*^-16}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"1.5529883223297643`*^-9"},
        {"1.1367368704148938`*^-11"},
        {"8.516872805742918`*^-14"},
        {"1.2879958874469441`*^-15"},
        {"1.770994345239548`*^-15"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{1.5529883223297643`*^-9, 1.1367368704148938`*^-11, 
       8.516872805742918*^-14, 1.2879958874469441`*^-15, 
       1.770994345239548*^-15}],
      Editable->False]},
    {
     RowBox[{"Log", "[", "x", "]"}], 
     InterpretationBox[GridBox[{
        {"6"},
        {"8"},
        {"10"},
        {"12"},
        {"14"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{6, 8, 10, 12, 14}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"9.688796143159095`*^-8"},
        {"7.569360693793215`*^-10"},
        {"5.912825784548659`*^-12"},
        {"2.2315482794965646`*^-14"},
        {"6.661338147750939`*^-16"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{9.688796143159095*^-8, 7.569360693793215*^-10, 
       5.912825784548659*^-12, 2.2315482794965646`*^-14, 
       6.661338147750939*^-16}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"0.`"},
        {"0.`"},
        {"0.`"},
        {"0.`"},
        {"0.`"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{0., 0., 0., 0., 0.}],
      Editable->False]},
    {
     SuperscriptBox["x", "26"], 
     InterpretationBox[GridBox[{
        {"6"},
        {"8"},
        {"10"},
        {"12"},
        {"14"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{6, 8, 10, 12, 14}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"5.80785419757035`*^-15"},
        {"5.620504062164855`*^-15"},
        {"1.3114509478384662`*^-15"},
        {"1.3114509478384662`*^-15"},
        {"1.3114509478384662`*^-15"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{5.80785419757035*^-15, 5.620504062164855*^-15, 
       1.3114509478384662`*^-15, 1.3114509478384662`*^-15, 
       1.3114509478384662`*^-15}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"1.906962088238373`*^-11"},
        {"4.683753385137379`*^-14"},
        {"0.`"},
        {"0.`"},
        {"0.`"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{1.906962088238373*^-11, 4.683753385137379*^-14, 0., 0., 
       0.}],
      Editable->False]},
    {
     FractionBox["1", 
      RowBox[{"1", "+", 
       RowBox[{"10000", " ", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           FractionBox["1", "2"], "-", "x"}], ")"}], "2"]}]}]], 
     InterpretationBox[GridBox[{
        {"6"},
        {"8"},
        {"10"},
        {"12"},
        {"14"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{6, 8, 10, 12, 14}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"1.7897597138057488`*^-15"},
        {"1.901619695918608`*^-15"},
        {"1.3423197853543115`*^-15"},
        {"1.3423197853543115`*^-15"},
        {"1.4541797674671707`*^-15"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{1.7897597138057488`*^-15, 1.901619695918608*^-15, 
       1.3423197853543115`*^-15, 1.3423197853543115`*^-15, 
       1.4541797674671707`*^-15}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"5.1618012465943144`*^-11"},
        {"7.94205873001301`*^-14"},
        {"2.237199642257186`*^-16"},
        {"1.118599821128593`*^-16"},
        {"2.237199642257186`*^-16"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{5.1618012465943144`*^-11, 7.94205873001301*^-14, 
       2.237199642257186*^-16, 1.118599821128593*^-16, 
       2.237199642257186*^-16}],
      Editable->False]},
    {
     RowBox[{"-", 
      FractionBox[
       RowBox[{
        SuperscriptBox["\[ExponentialE]", 
         RowBox[{"1", "-", 
          FractionBox["1", 
           RowBox[{"1", "-", "x"}]]}]], " ", 
        RowBox[{"Sin", "[", 
         RowBox[{"1", "-", 
          FractionBox["1", 
           RowBox[{"1", "-", "x"}]]}], "]"}]}], 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"1", "-", "x"}], ")"}], "2"]]}], 
     InterpretationBox[GridBox[{
        {"6"},
        {"8"},
        {"10"},
        {"12"},
        {"14"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{6, 8, 10, 12, 14}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"1.2414136385530128`*^-10"},
        {"1.7763568394002505`*^-15"},
        {"1.7763568394002505`*^-15"},
        {"1.7763568394002505`*^-15"},
        {"1.1102230246251565`*^-15"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{1.2414136385530128`*^-10, 1.7763568394002505`*^-15, 
       1.7763568394002505`*^-15, 1.7763568394002505`*^-15, 
       1.1102230246251565`*^-15}],
      Editable->False], 
     InterpretationBox[GridBox[{
        {"1.475264355121908`*^-12"},
        {"1.8345325258906087`*^-12"},
        {"1.8463008899516353`*^-12"},
        {"0.`"},
        {"0.`"}
       },
       BaselinePosition->{Baseline, {1, 1}},
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}}],
      ColumnForm[{1.475264355121908*^-12, 1.8345325258906087`*^-12, 
       1.8463008899516353`*^-12, 0., 0.}],
      Editable->False]}
   },
   GridBoxFrame->{"Columns" -> {{True}}, "Rows" -> {{True}}}],
  "Grid"]], "Output",
 ImageSize->{457, 697},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[73]=",
 CellID->6014066]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Singularity Handling", "Section",
 CellTags->"SingularityHandling",
 CellID->122144792],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s adaptive strategies speed up their convergence through variable \
transformations at the integration region boundaries and user-specified \
singular points or manifolds. ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s adaptive strategies also ignore the integrand evaluation results at \
singular points."
}], "Text",
 CellID->410026393],

Cell[TextData[{
 "Singularity specification is discussed in ",
 ButtonBox["User specified singularities",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#112801667"],
 "."
}], "Text",
 CellID->49287082],

Cell[TextData[{
 "For ",
 "multidimensional",
 " singularity handling with variable transformations should be used with \
caution; see ",
 ButtonBox["IMT multidimensional singularity handling",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#159907828"],
 ". Coordinate change for a multidimensional integral can simplify or \
eliminate singularities; see ",
 ButtonBox["Duffy's coordinates for multidimensional singularity handling",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#738848244"],
 "."
}], "Text",
 CellID->570216418],

Cell[TextData[{
 "For details about how ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " ignores singularities see ",
 ButtonBox["Ignoring the singularity",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#74051012"],
 "."
}], "Text",
 CellID->713387871],

Cell[TextData[{
 "The computation of Cauchy principle value integrals is described in ",
 ButtonBox["Cauchy principal value integration",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#295594003"],
 "."
}], "Text",
 CellID->289686150],

Cell[CellGroupData[{

Cell["User-specified singularities", "Subsection",
 CellTags->"UserSingularities",
 CellID->112801667],

Cell[CellGroupData[{

Cell["Point singularities", "Subsubsection",
 CellID->19203486],

Cell[TextData[{
 "If it is known where the singularities occur, they can be specified in the \
ranges of integration, or through the option ",
 Cell[BoxData[
  ButtonBox["Exclusions",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exclusions"]], "InlineFormula"],
 "."
}], "Text",
 CellID->156529646],

Cell[TextData[{
 "Here is an example of an integral that has two singular points at ",
 Cell[BoxData[
  FormBox[
   FractionBox["\[Pi]", "6"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   FractionBox["\[Pi]", "3"], TraditionalForm]]],
 "."
}], "MathCaption",
 CellID->106906093],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{"(", 
       RowBox[{"x", "-", 
        FractionBox["\[Pi]", "6"]}], ")"}]]], 
    FractionBox["1", 
     SqrtBox[
      RowBox[{"(", 
       RowBox[{"x", "-", 
        FractionBox["\[Pi]", "3"]}], ")"}]]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     FractionBox["\[Pi]", "6"], ",", 
     FractionBox["\[Pi]", "3"], ",", 
     FractionBox["\[Pi]", "2"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[58]:=",
 CellID->19599620],

Cell[BoxData[
 RowBox[{"3.874437958831575`*^-8", "-", 
  RowBox[{"3.141592636647447`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{162, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[58]=",
 CellID->373100846]
}, Open  ]],

Cell[TextData[{
 "Here is an example of a two-dimensional integral with a singular point at \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"1", ",", "1"}], ")"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->545875582],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "x"}], ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "y"}], ")"}], "2"]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[59]:=",
 CellID->364804001],

Cell[BoxData[
 RowBox[{"-", "2.9442259631127135`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[59]=",
 CellID->388907584]
}, Open  ]],

Cell[TextData[{
 "Here is an example of an integral that has two singular points at ",
 Cell[BoxData[
  FormBox[
   FractionBox["\[Pi]", "6"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   FractionBox["\[Pi]", "3"], TraditionalForm]], "InlineMath"],
 " specified with the ",
 Cell[BoxData[
  ButtonBox["Exclusions",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exclusions"]], "InlineFormula"],
 " option."
}], "MathCaption",
 CellID->1158717],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{"(", 
       RowBox[{"x", "-", 
        FractionBox["\[Pi]", "6"]}], ")"}]]], 
    FractionBox["1", 
     SqrtBox[
      RowBox[{"(", 
       RowBox[{"x", "-", 
        FractionBox["\[Pi]", "3"]}], ")"}]]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     FractionBox["\[Pi]", "2"]}], "}"}], ",", 
   RowBox[{"Exclusions", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      FractionBox["\[Pi]", "6"], ",", 
      FractionBox["\[Pi]", "3"]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[60]:=",
 CellID->438906078],

Cell[BoxData[
 RowBox[{"3.874437958831575`*^-8", "-", 
  RowBox[{"3.141592636647447`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{162, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[60]=",
 CellID->9356815]
}, Open  ]],

Cell[TextData[{
 "Here is an example of a two-dimensional integral with a singular point at \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"1", ",", "1"}], ")"}], TraditionalForm]], "InlineMath"],
 " specified with the ",
 Cell[BoxData[
  ButtonBox["Exclusions",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exclusions"]], "InlineFormula"],
 " option."
}], "MathCaption",
 CellID->251045976],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "x"}], ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "y"}], ")"}], "2"]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"Exclusions", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[61]:=",
 CellID->18458950],

Cell[BoxData[
 RowBox[{"-", "2.9442259631127135`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[61]=",
 CellID->354989340]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Curve, surface, and hypersurface singularities", "Subsubsection",
 CellID->146947120],

Cell[TextData[{
 "Singularities over curves, surfaces, or hypersurfaces in general can be \
specified through the option ",
 Cell[BoxData[
  ButtonBox["Exclusions",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exclusions"]], "InlineFormula"],
 " with their equations. Such singularities, generally, cannot be specified \
using ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s variable ranges. "
}], "Text",
 CellID->144923299],

Cell[TextData[{
 "This ",
 "two-dimensional",
 " function is singular along the curve ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "2"], "+", 
     SuperscriptBox["y", "2"]}], "=", "1"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->38793013],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"x", "^", "2"}], "+", 
         RowBox[{"y", "^", "2"}]}], ")"}]}], ")"}], "^", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"Exclusions", "\[Rule]", 
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], "\[Equal]", "1"}]}]}], "]"}]], "Input",
 CellLabel->"In[62]:=",
 CellID->6237669],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 142},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[62]=",
 CellID->1034508636]
}, Open  ]],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  ButtonBox["Exclusions",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exclusions"]], "InlineFormula"],
 " the integral is quickly calculated."
}], "MathCaption",
 CellID->323771085],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Log", "[", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"x", "^", "2"}], "+", 
          RowBox[{"y", "^", "2"}]}], ")"}]}], ")"}], "^", "2"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "2"}], "}"}], ",", 
    RowBox[{"Exclusions", "\[Rule]", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], "\[Equal]", "1"}]}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[12]:=",
 CellID->92332513],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.33294999999999886`", ",", "1.2813158950133732`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]=",
 CellID->453056113]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " will reach convergence much more slowly if no singularity specification is \
given."
}], "MathCaption",
 CellID->154464628],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"Log", "[", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"x", "^", "2"}], "+", 
          RowBox[{"y", "^", "2"}]}], ")"}]}], ")"}], "^", "2"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "2"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[35]:=",
 CellID->3158110],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[35]:=",
 CellID->468616535],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"eincr\"\>"}], ":", 
  " ", "\<\"The global error of the strategy GlobalAdaptive has increased \
more than \\!\\(2000\\) times. The global error is expected to decrease \
monotonically after a number of integrand evaluations. Suspect one of the \
following: the working precision is insufficient for the specified precision \
goal; the integrand is highly oscillatory or it is not a (piecewise) smooth \
function; or the true value of the integral is 0. Increasing the value of the \
GlobalAdaptive option MaxErrorIncreases might lead to a convergent numerical \
integration. NIntegrate obtained \\!\\(1.2814579311938816`\\) and \
\\!\\(0.0003423677128377028`\\) for the integral and error estimates. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NIntegrate/eincr\\\", \
ButtonNote -> \\\"NIntegrate::eincr\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[35]:=",
 CellID->1215570196],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.4347759999999994`", ",", "1.2814579311938816`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]=",
 CellID->1076350261]
}, Open  ]],

Cell[TextData[{
 "Here is an example of a case in which a singular curve ",
 StyleBox["can ",
  FontSlant->"Italic"],
 "be specified with the variable ranges. If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "\[Element]", 
    RowBox[{"[", 
     RowBox[{"0", ",", "2"}], "]"}]}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[Element]", 
    RowBox[{"[", 
     RowBox[{"0", ",", "2"}], "]"}]}], TraditionalForm]], "InlineMath"],
 " this would not be possible\[LongDash]see the following example."
}], "MathCaption",
 CellID->134709425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"x", "^", "2"}], "+", 
         RowBox[{"y", "^", "2"}]}], ")"}]}], ")"}], "^", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", 
     SqrtBox[
      RowBox[{"1", "-", 
       SuperscriptBox["x", "2"]}]], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->143201609],

Cell[BoxData[
 RowBox[{"-", "2.336137615336383`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->11582058]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Example implementation of curve, surface, and hypersurface singularity \
handling\
\>", "Subsubsection",
 CellID->542398406],

Cell[TextData[{
 "If the curve, surface, or hypersurface on which the singularities occur is \
known in implicit form (i.e. it can be described as a single equation) the \
function ",
 Cell[BoxData[
  ButtonBox["Boole",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Boole"]], "InlineFormula"],
 " can be used to form integration regions that have the singular curves, \
surfaces, or hypersurfaces as boundaries. "
}], "Text",
 CellID->156695069],

Cell[TextData[{
 "This two-dimensional function has singular points along the curve ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SuperscriptBox["x", "2"], "+", 
     SuperscriptBox["y", "2"]}], "=", "1"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->33513663],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"x", "^", "2"}], "+", 
         RowBox[{"y", "^", "2"}]}], ")"}]}], ")"}], "^", "2"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"Exclusions", "\[Rule]", 
    RowBox[{
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], "\[Equal]", "1"}]}]}], "]"}]], "Input",
 CellLabel->"In[66]:=",
 CellID->440660927],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 142},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[66]=",
 CellID->382101928]
}, Open  ]],

Cell[TextData[{
 "Using ",
 Cell[BoxData[
  ButtonBox["Boole",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Boole"]], "InlineFormula"],
 " the integral is calculated quickly."
}], "MathCaption",
 CellID->174124104],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Log", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"x", "^", "2"}], "+", 
            RowBox[{"y", "^", "2"}]}], ")"}]}], ")"}], "^", "2"}], "]"}], "*",
       
      RowBox[{"Boole", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"x", "^", "2"}], "+", 
         RowBox[{"y", "^", "2"}]}], "<", "1"}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "2"}], "}"}]}], "]"}], "+", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Log", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"x", "^", "2"}], "+", 
            RowBox[{"y", "^", "2"}]}], ")"}]}], ")"}], "^", "2"}], "]"}], "*",
       
      RowBox[{"Boole", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"x", "^", "2"}], "+", 
         RowBox[{"y", "^", "2"}]}], ">", "1"}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "2"}], "}"}]}], "]"}]}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->453484095],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.29595499999999975`", ",", "1.2813158950133732`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->630976836]
}, Open  ]],

Cell[TextData[{
 "This two-dimensional function has singular points along the curve ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"x", "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "y"}], ")"}], "2"]}], "=", "1"}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->78459075],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", 
       RowBox[{"(", 
        RowBox[{"x", "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"1", "-", "y"}], ")"}], "2"]}], ")"}]}], ")"}], "2"], 
    "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
   RowBox[{"Exclusions", "\[Rule]", " ", 
    RowBox[{
     RowBox[{"x", "+", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"1", "-", "y"}], ")"}], "2"]}], "\[Equal]", "1"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[68]:=",
 CellID->209370751],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 152},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[68]=",
 CellID->606068278]
}, Open  ]],

Cell[TextData[{
 "Again, using ",
 Cell[BoxData[
  ButtonBox["Boole",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Boole"]], "InlineFormula"],
 " the integral is calculated quickly."
}], "MathCaption",
 CellID->149901165],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Log", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{"(", 
           RowBox[{"x", "+", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], ")"}]}], ")"}], 
        "^", "2"}], "]"}], "*", 
      RowBox[{"Boole", "[", 
       RowBox[{
        RowBox[{"x", "+", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], "<", "1"}], "]"}]}], 
     ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
     RowBox[{"PrecisionGoal", "\[Rule]", "4"}]}], "]"}], "+", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Log", "[", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"1", "-", 
          RowBox[{"(", 
           RowBox[{"x", "+", 
            RowBox[{
             RowBox[{"(", 
              RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], ")"}]}], ")"}], 
        "^", "2"}], "]"}], "*", 
      RowBox[{"Boole", "[", 
       RowBox[{
        RowBox[{"x", "+", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], ">", "1"}], "]"}]}], 
     ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
     RowBox[{"PrecisionGoal", "\[Rule]", "4"}]}], "]"}]}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->22085070],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.43293300000000007`", ",", 
   RowBox[{"-", "2.222428625017353`"}]}], "}"}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->304879365]
}, Open  ]],

Cell["This is how the sampling points of the integration look. ", \
"MathCaption",
 CellID->15871014],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"gr1", "=", 
   RowBox[{"{", 
    RowBox[{"Red", ",", 
     RowBox[{"Point", "/@", 
      RowBox[{"N", "@", 
       RowBox[{
        RowBox[{"Reap", "[", 
         RowBox[{"NIntegrate", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"Log", "[", 
             RowBox[{
              RowBox[{"(", 
               RowBox[{"1", "-", 
                RowBox[{"(", 
                 RowBox[{"x", "+", 
                  RowBox[{
                   RowBox[{"(", 
                    RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], ")"}]}], 
               ")"}], "^", "2"}], "]"}], "*", 
            RowBox[{"Boole", "[", 
             RowBox[{
              RowBox[{"x", "+", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], "<", "1"}], 
             "]"}]}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", 
             RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"y", ",", 
             RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
           RowBox[{"PrecisionGoal", "\[Rule]", "4"}], ",", 
           RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
            RowBox[{"Sow", "[", 
             RowBox[{"{", 
              RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
        RowBox[{"[", 
         RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}]}], "}"}]}], ";", 
  RowBox[{"gr2", "=", 
   RowBox[{"{", 
    RowBox[{"Blue", ",", 
     RowBox[{"Point", "/@", 
      RowBox[{"N", "@", 
       RowBox[{
        RowBox[{"Reap", "[", 
         RowBox[{"NIntegrate", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"Log", "[", 
             RowBox[{
              RowBox[{"(", 
               RowBox[{"1", "-", 
                RowBox[{"(", 
                 RowBox[{"x", "+", 
                  RowBox[{
                   RowBox[{"(", 
                    RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], ")"}]}], 
               ")"}], "^", "2"}], "]"}], "*", 
            RowBox[{"Boole", "[", 
             RowBox[{
              RowBox[{"x", "+", 
               RowBox[{
                RowBox[{"(", 
                 RowBox[{"1", "-", "y"}], ")"}], "^", "2"}]}], ">", "1"}], 
             "]"}]}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", 
             RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"y", ",", 
             RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
           RowBox[{"PrecisionGoal", "\[Rule]", "4"}], ",", 
           RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
            RowBox[{"Sow", "[", 
             RowBox[{"{", 
              RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
        RowBox[{"[", 
         RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}]}], "}"}]}], ";"}], "\n", 
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"PointSize", "[", "0.006", "]"}], ",", "gr1", ",", "gr2"}], 
    "}"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}], ",", 
   RowBox[{"AxesOrigin", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "2"}], ",", 
      RowBox[{"-", "1"}]}], "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[6]:=",
 CellID->58505441],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 183},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->151823548]
}, Open  ]],

Cell[TextData[{
 "Here is a function that takes a singular curve, surface, or hypersurface \
specification and uses the function ",
 Cell[BoxData[
  ButtonBox["Boole",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Boole"]], "InlineFormula"],
 " to make integration regions that have the singularities on their \
boundaries. "
}], "MathCaption",
 CellID->627364560],

Cell[BoxData[
 RowBox[{
  RowBox[{"SingularManifoldNIntegrate", "[", 
   RowBox[{"f_", ",", "ranges___", ",", 
    RowBox[{"Equal", "[", 
     RowBox[{"eq_", ",", 
      RowBox[{"n_", "?", "NumericQ"}]}], "]"}], ",", "opts___"}], "]"}], ":=",
   "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"f", "*", 
      RowBox[{"Boole", "[", 
       RowBox[{"eq", "<", "n"}], "]"}]}], ",", "ranges", ",", "opts"}], "]"}],
    "+", 
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"f", "*", 
      RowBox[{"Boole", "[", 
       RowBox[{"eq", ">", "n"}], "]"}]}], ",", "ranges", ",", "opts"}], 
    "]"}]}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->116160749],

Cell["This defines a three-dimensional function.", "MathCaption",
 CellID->6637831],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x_", ",", "y_", ",", "z_"}], "]"}], ":=", 
   RowBox[{"Log", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"(", 
       RowBox[{"1", "-", 
        RowBox[{"(", 
         RowBox[{"x", "+", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{"1", "-", "y"}], ")"}], "^", "2"}], "+", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{"1", "-", "z"}], ")"}], "^", "2"}]}], ")"}]}], ")"}], 
      ")"}], "^", "2"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->629712713],

Cell[TextData[{
 "Here is the integral of a three-dimensional function with singular points \
along the surface ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"x", "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "y"}], ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "z"}], ")"}], "2"]}], "=", "1"}], TraditionalForm]], 
  "InlineMath"],
 ". "
}], "MathCaption",
 CellID->84228831],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"SingularManifoldNIntegrate", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{
    RowBox[{"x", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "y"}], ")"}], "^", "2"}], "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "z"}], ")"}], "^", "2"}]}], "\[Equal]", "1"}], ",", 
   RowBox[{"PrecisionGoal", "\[Rule]", "3"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->349557344],

Cell[BoxData[
 RowBox[{"21.747126121694514`", "\[InvisibleSpace]", "-", 
  RowBox[{
  "4.892636912996955068736973046`15.954589770191005*^-339", " ", 
   "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{242, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->333754661]
}, Open  ]],

Cell["These are the sampling points of the integration. ", "MathCaption",
 CellID->313643860],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"gr1", "=", 
   RowBox[{"{", 
    RowBox[{"Red", ",", 
     RowBox[{
      RowBox[{
       RowBox[{"Point", "[", 
        RowBox[{"Re", "[", "#", "]"}], "]"}], "&"}], "/@", 
      RowBox[{
       RowBox[{"Reap", "[", 
        RowBox[{"NIntegrate", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"f", "[", 
            RowBox[{"x", ",", "y", ",", "z"}], "]"}], "*", 
           RowBox[{"Boole", "[", 
            RowBox[{
             RowBox[{"x", "+", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"1", "-", "y"}], ")"}], "^", "2"}], "+", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"1", "-", "z"}], ")"}], "^", "2"}]}], "<", "1"}], 
            "]"}]}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", 
            RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"y", ",", 
            RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"z", ",", 
            RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
          RowBox[{"PrecisionGoal", "\[Rule]", "3"}], ",", 
          RowBox[{"EvaluationMonitor", ":>", 
           RowBox[{"Sow", "[", 
            RowBox[{"{", 
             RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]}]}], "]"}], 
        "]"}], "[", 
       RowBox[{"[", 
        RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}], "}"}]}], ";", 
  RowBox[{"gr2", "=", 
   RowBox[{"{", 
    RowBox[{"Blue", ",", 
     RowBox[{
      RowBox[{
       RowBox[{"Point", "[", 
        RowBox[{"Re", "[", "#", "]"}], "]"}], "&"}], "/@", 
      RowBox[{
       RowBox[{"Reap", "[", 
        RowBox[{"NIntegrate", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"f", "[", 
            RowBox[{"x", ",", "y", ",", "z"}], "]"}], "*", 
           RowBox[{"Boole", "[", 
            RowBox[{
             RowBox[{"x", "+", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"1", "-", "y"}], ")"}], "^", "2"}], "+", 
              RowBox[{
               RowBox[{"(", 
                RowBox[{"1", "-", "z"}], ")"}], "^", "2"}]}], ">", "1"}], 
            "]"}]}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", 
            RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"y", ",", 
            RowBox[{"-", "1"}], ",", "3"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"z", ",", 
            RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
          RowBox[{"PrecisionGoal", "\[Rule]", "3"}], ",", 
          RowBox[{"EvaluationMonitor", ":>", 
           RowBox[{"Sow", "[", 
            RowBox[{"{", 
             RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]}]}], "]"}], 
        "]"}], "[", 
       RowBox[{"[", 
        RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}], "}"}]}], ";"}], "\n", 
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"PointSize", "[", "0.006", "]"}], ",", "gr1", ",", "gr2"}], 
    "}"}], ",", 
   RowBox[{"Axes", "->", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[4]:=",
 CellID->227393257],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 310},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->734898644]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"SingularityHandler\" and \"SingularityDepth\"", "Subsection",
 CellID->31583791],

Cell[TextData[{
 "Adaptive strategies improve the integral estimate by region bisection. If \
an adaptive strategy subregion is obtained by the number of bisections \
specified by the option ",
 Cell[BoxData["\"\<SingularityDepth\>\""], "InlineFormula"],
 ", it is decided that subregion has a singularity. Then the integration over \
that subregion is done with the singularity handler specified by ",
 Cell[BoxData["\"\<SingularityHandler\>\""], "InlineFormula"],
 "."
}], "Text",
 CellID->601971160],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<SingularityDepth\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
number of recursive bisections before applying a singularity handler\
\>", "TableText"]},
   {"\"\<SingularityHandler\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "singularity handler", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->222155288],

Cell[TextData[{
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " singularity handling options."
}], "Caption",
 CellID->103638349],

Cell[TextData[{
 "If there is an integrable singularity at the boundary of a given region of \
integration, bisection could easily recur to ",
 StyleBox[ButtonBox["MaxRecursion",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#280704764"], 
  "InlineCode"],
 " before convergence occurs. To deal with these situations ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s adaptive strategies use variable transformations (",
 StyleBox[ButtonBox["IMT",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#150310618"], 
  "InlineCode"],
 ", ",
 StyleBox["\"", "InlineCode"],
 StyleBox[ButtonBox["DoubleExponential\"",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#225088061"], 
  "InlineCode"],
 ", ",
 Cell[BoxData["SidiSin"], "InlineFormula"],
 "), to speed up the integration convergence, or a region transformation (",
 ButtonBox["Duffy's coordinates",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#738848244"],
 ") that relaxes the order of the singularity. The theoretical background of \
the variable transformation singularity handlers is given by the \
Euler-Maclaurin formula [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "]."
}], "Text",
 CellID->189524672]
}, Open  ]],

Cell[CellGroupData[{

Cell["Use of the IMT variable transformation", "Subsection",
 CellTags->"IMTrule",
 CellID->150310618],

Cell[TextData[{
 "The IMT variable transformation is the variable transformation in a \
quadrature method proposed by Iri, Moriguti, and Takasawa, called in the \
literature the ",
 StyleBox["IMT rule",
  FontSlant->"Italic"],
 " [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "][",
 ButtonBox["IriMorTak70",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#18838335"],
 "]. The IMT rule",
 StyleBox[" ",
  FontSlant->"Italic"],
 "is based upon the idea of transforming the independent variable in such a \
way that all derivatives of the new integrand vanish at the end points of the \
integration interval. A trapezoidal rule is then applied to the new integrand \
and under proper conditions high accuracy of the result might be attained [",
 ButtonBox["IriMorTak70",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#18838335"],
 "][",
 ButtonBox["Mori74",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#137456387"],
 "]. "
}], "Text",
 CellID->10156937],

Cell["\<\
Here is a numerical integration that uses the IMT variable transformation for \
singularity handling.\
\>", "Text",
 CellID->84485055],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"1", "-", "x"}], "]"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
      RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"IMT", ",", " ", 
         RowBox[{"\"\<TuningParameters\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"10", ",", "2"}], "}"}]}]}], "}"}]}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->158276731],

Cell[BoxData["1.9999999999910048`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->150771452]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<TuningParameters\>\"", "10", Cell[TextData[{
     "a pair of numbers ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        StyleBox["a", "TI"], ",", 
        StyleBox["p", "TI"]}], "}"}]], "InlineFormula"],
     " that are the tuning parameters in the IMT transformation formula ",
     Cell[BoxData[
      RowBox[{
       StyleBox["a", "TI"], " ", 
       SuperscriptBox["\[ExponentialE]", 
        RowBox[{"1", "-", 
         FractionBox["1", 
          SuperscriptBox[
           StyleBox["t", "TI"], 
           StyleBox["p", "TI"]]]}]]}]], "InlineFormula"],
     "; if only a number ",
     Cell[BoxData[
      StyleBox["a", "TI"]], "InlineFormula"],
     " is given, it is interpreted as ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        StyleBox["a", "TI"], ",", "1"}], "}"}]], "InlineFormula"]
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->189824841],

Cell[TextData[{
 Cell[BoxData["IMT"], "InlineFormula"],
 " singularity handler options."
}], "Caption",
 CellID->168671720],

Cell[TextData[{
 "Adaptive strategies of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " employ only the transformation of",
 StyleBox[" ",
  FontSlant->"Italic"],
 "the",
 StyleBox[" ",
  FontSlant->"Italic"],
 "IMT rule. With the decision that a region might have a singularity, the IMT \
transformation is applied to its integrand. The integration continues, \
though, not with a trapezoidal rule, but with the same integration rule used \
before the transformation. (Singularity handling with ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " switches to trapezoidal integration rule.)"
}], "Text",
 CellID->23765367],

Cell[TextData[{
 "Also, adaptive strategies of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " use a variant of the original IMT transformation, with the transformed \
integrand vanishing only at one of the ends."
}], "Text",
 CellID->219675416],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s IMT transformation ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["\[CurlyPhi]", 
       RowBox[{"a", ",", "p"}]], "(", "t", ")"}], ":", 
     RowBox[{"(", 
      RowBox[{"0", ",", "1"}], "]"}]}], "\[Rule]", 
    RowBox[{"(", 
     RowBox[{"0", ",", "1"}], "]"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"a", ">", "0"}], ",", 
    RowBox[{"p", ">", "0"}], ","}], TraditionalForm]], "InlineMath"],
 " is defined."
}], "Text",
 CellID->374374527],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"\[CurlyPhi]", "[", 
    RowBox[{"a_", ",", "p_", ",", "t_"}], "]"}], ":=", 
   RowBox[{"a", " ", 
    RowBox[{"Exp", "[", 
     RowBox[{"1", "-", 
      FractionBox["1", 
       SuperscriptBox["t", "p"]]}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"\[CurlyPhi]", "[", "t_", "]"}], ":=", 
  RowBox[{"\[CurlyPhi]", "[", 
   RowBox[{"1", ",", "1", ",", "t"}], "]"}]}]}], "Input",
 CellLabel->"In[14]:=",
 CellID->383398155],

Cell[TextData[{
 "The parameters ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " are called tuning parameters [",
 ButtonBox["MurIri82",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#151206575"],
 "]. "
}], "Text",
 CellID->1716844],

Cell[TextData[{
 "The limit of the derivative of the IMT transformation is ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->205161309],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Limit", "[", 
  RowBox[{
   RowBox[{"D", "[", 
    RowBox[{
     RowBox[{"\[CurlyPhi]", "[", 
      RowBox[{"a", ",", "p", ",", "t"}], "]"}], ",", "t"}], "]"}], ",", 
   RowBox[{"t", "\[Rule]", "0"}], ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"a", ">", "0"}], ",", 
      RowBox[{"p", ">", "0"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[16]:=",
 CellID->62572874],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[16]=",
 CellID->19580145]
}, Open  ]],

Cell["Here is the plot of the IMT transformation.", "Text",
 CellID->321316718],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"\[CurlyPhi]", "[", "t", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"AxesOrigin", "->", 
    RowBox[{"{", 
     RowBox[{"0", ",", 
      RowBox[{"-", "0.02"}]}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"AspectRatio", "\[Rule]", "Automatic"}]}], "]"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->476801895],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{187, 188},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->249792560]
}, Open  ]],

Cell["\<\
From the graph above follows that the transformed sampling points are much \
denser around 0. This means that if the integrand is singular at 0 it will be \
sampled more effectively, since a larger part of the integration rule \
sampling points will contribute large integrand values to the integration \
rule's integral estimate.\
\>", "Text",
 CellID->467636680],

Cell[TextData[{
 "Since for any given working precision the numbers around 0 are much denser \
than the numbers around 1, after a region bisection ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s adaptive strategies reverse the bisection variable of the subregion that \
has the right end of the bisected interval. This can be seen from the \
following plot."
}], "Text",
 CellID->12661134],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pnts", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        SqrtBox["x"]], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
          RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"IMT", ",", 
             RowBox[{"\"\<TuningParameters\>\"", "\[Rule]", "1"}]}], 
            "}"}]}]}], "}"}]}], ",", 
       RowBox[{"PrecisionGoal", "\[Rule]", "2"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], 
    "\[LeftDoubleBracket]", 
    RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"Transpose", "[", 
   RowBox[{"{", 
    RowBox[{"pnts", ",", 
     RowBox[{"Range", "[", 
      RowBox[{"Length", "[", "pnts", "]"}], "]"}]}], "}"}], "]"}], 
  "]"}]}], "Input",
 CellLabel->"In[18]:=",
 CellID->378272796],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJw1lFtIk3Echj8qQorIQCqiCw2NihGZZlrpfuo8H9rJTWdRGVldmFqhdeVX
IhWhphXaYSQVFR0syVQqa2KhlOnSjhpaWBCdDIIOF533/IUxHl5efPf8vi0o
r9C6YZymaVv+vv69e1yNGz+m6Ubt/99no8eQ/7N+wjlYE7nf/7Wk9DbsJ99r
eia07XsA+4s2y7/CNOUpHCC+95fwTNG+lLav3vkani27h3p+1bjfwoFSfyYk
aNq0D/AcCXfPL2op+wQHy+HEdaneW2PwXPHtLYLniW9vPbyAvZdhA3vb4YXs
7YIXsdcLh7L3EbyYvc/gMPa+gMPZOwwvYe8IHCGppxaFbBpR+5fK0d6AyCGj
4khJH6g9sNdPfd4o6SjI/92dqXwsE/3HjeHGFe/g5exV/laIPtEUZOl8A0eL
sfaItrRpFI6RZkNBYem3V7BR8kPfdwV71H1EWlb1DdYFwrrgdy15LH73ksfi
9wR5HH4vkcfht5U8nr0e8nj83iE34beb3ITfHvIE/PaSJ+C3jzwRv/jWE/Gr
/Cfhl/voSfhV90vG7yB5MnvVvVPw+5g8Bb8D5Kn4fUieil/1PKXhF9bT8Gsm
T8fvDvJ0/FaTZ+DXTZ6B37Pkmey9Qp6J32bylfhtI1+J35vkZvze8rGY8cv3
XTfjF98eM36Vfwt+uY9Y8Mv9dAt+79G3sJd7a1b83qVvxW8nfSt+O+hb8cvz
pNnwC4sNv0Lfht9N9G34LaNvx28lfTt+6+jb2dtA347f0/Sz8MvvpWTh9yL9
LPw20s/Cr/p9cuAX3+LAr/LvwC/38Tjwq+7nxO91+k72qns78XuNvhO/V+ln
47eJfjZ+1fOUjV/Yky3RrY2HRo8Xh/v6ObK9crr1SdfzaF8/R05sC9tTPGMS
/Rw5tnV6TEV8Hv0cKT+4qyjwfAX/3yWRY4YIx358i0vGN1dNrYo6Sd8ltvUN
7v5mnnePSwLKve473gv0c2XymqTN1SX4lVzRJznbKpVfPVf+ADt96Ww=
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 0.9960213400237106}, {0., 110.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->24189893]
}, Open  ]],

Cell["\<\
No other singularity handler is applied to the subregions of a region to \
which the IMT variable transformation has been applied.\
\>", "Text",
 CellID->964496099]
}, Open  ]],

Cell[CellGroupData[{

Cell["IMT transformation by example", "Subsection",
 CellID->107913531],

Cell[TextData[{
 "Consider the function ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", 
    SqrtBox["x"]], TraditionalForm]], "InlineMath"],
 " over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"0", ",", "1"}], "]"}], TraditionalForm]], "InlineMath"],
 " that has a singularity at ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->504815874],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  RowBox[{"1", "/", 
   RowBox[{"Sqrt", "[", "x", "]"}]}]}]], "Input",
 CellLabel->"In[29]:=",
 CellID->215161628],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[30]:=",
 CellID->25392041],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{263, 167},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->15745112]
}, Open  ]],

Cell[TextData[{
 "Assume the integration is done with ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 StyleBox[",", "InlineCode"],
 " with singularity handler IMT and singularity depth 4. After 4 bisections \
",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " will have a region with boundaries ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"1", "/", "16"}]}], "}"}], TraditionalForm]], "InlineMath"],
 " that contains the singular end point. For that region the IMT variable \
transformation will change its boundaries to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}], TraditionalForm]], "InlineMath"],
 " and its integrand to the following."
}], "Text",
 CellID->120084425],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"1", "/", "16"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"f", "[", 
   RowBox[{"Rescale", "[", 
    RowBox[{
     RowBox[{"\[CurlyPhi]", "[", "t", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"a", ",", "b"}], "}"}]}], "]"}], "]"}], 
  RowBox[{"D", "[", 
   RowBox[{
    RowBox[{"Rescale", "[", 
     RowBox[{
      RowBox[{"\[CurlyPhi]", "[", "t", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"a", ",", "b"}], "}"}]}], "]"}], ",", "t"}], "]"}]}]}], "Input",\

 CellLabel->"In[31]:=",
 CellID->196201412],

Cell[BoxData[
 FractionBox[
  SqrtBox[
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"1", "-", 
     FractionBox["1", "t"]}]]], 
  RowBox[{"4", " ", 
   SuperscriptBox["t", "2"]}]]], "Output",
 ImageSize->{54, 57},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->116365017]
}, Open  ]],

Cell["Here is the plot of the new integrand.", "Text",
 CellID->318247493],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"1", "/", "16"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"Rescale", "[", 
       RowBox[{
        RowBox[{"\[CurlyPhi]", "[", "t", "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"a", ",", "b"}], "}"}]}], "]"}], "]"}], 
     RowBox[{"D", "[", 
      RowBox[{
       RowBox[{"Rescale", "[", 
        RowBox[{
         RowBox[{"\[CurlyPhi]", "[", "t", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"a", ",", "b"}], "}"}]}], "]"}], ",", "t"}], "]"}]}], "//", 
    "Evaluate"}], ",", 
   RowBox[{"{", 
    RowBox[{"t", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"AxesOrigin", "->", 
    RowBox[{"{", 
     RowBox[{"0", ",", 
      RowBox[{"-", "0.02"}]}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "Input",
 CellLabel->"In[33]:=",
 CellID->212391764],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{230, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]=",
 CellID->612786281]
}, Open  ]],

Cell["The singularity is smashed!", "Text",
 CellID->724856187],

Cell[TextData[{
 "Some of the sampling points, though, become too close to the singular end, \
and therefore special care should be taken for sampling points that coincide \
with the singular point because of the IMT transformation. ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " ignores evaluations at singular points; see ",
 ButtonBox["Ignoring the singularity",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#74051012"],
 "."
}], "Text",
 CellID->456093879],

Cell[TextData[{
 "For example, consider sampling points and weights of the ",
 ButtonBox["Gauss\[Dash]Kronrod rule.",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#381359375"]
}], "Text",
 CellID->28975310],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"absc", ",", "weights", ",", "errweight"}], "}"}], "=", 
   RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
    RowBox[{"5", ",", "MachinePrecision"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[35]:=",
 CellID->276704051],

Cell[TextData[{
 "The Gauss\[Dash]Kronrod sampling points for the region ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"0", ",", 
     RowBox[{"1", "/", "16"}]}], "}"}], TraditionalForm]], "InlineMath"],
 " and the derivatives of the rescaling follow."
}], "Text",
 CellID->114832530],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"abscGK", "=", 
  RowBox[{
   RowBox[{
    RowBox[{"Rescale", "[", 
     RowBox[{"#1", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", 
        FractionBox["1", "16"]}], "}"}]}], "]"}], "&"}], "/@", 
   "absc"}]}]], "Input",
 CellLabel->"In[36]:=",
 CellID->723506142],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.0004973324970361723`", ",", "0.0029318798144167546`", ",", 
   "0.007682289794660957`", ",", "0.014422834059197406`", ",", 
   "0.02251154958869428`", ",", "0.03125`", ",", "0.03998845041130572`", ",", 
   "0.048077165940802594`", ",", "0.05481771020533904`", ",", 
   "0.059568120185583245`", ",", "0.06200266750296383`"}], "}"}]], "Output",
 ImageSize->{438, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[36]=",
 CellID->353016476]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"derivativesGK", "=", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"D", "[", 
      RowBox[{
       RowBox[{"Rescale", "[", 
        RowBox[{"t", ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", 
           FractionBox["1", "16"]}], "}"}]}], "]"}], ",", "t"}], "]"}], "/.", 
     
     RowBox[{"t", "\[Rule]", "#"}]}], "&"}], "/@", "absc"}]}]], "Input",
 CellLabel->"In[37]:=",
 CellID->438887639],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"], ",", 
   FractionBox["1", "16"]}], "}"}]], "Output",
 ImageSize->{330, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[37]=",
 CellID->216808106]
}, Open  ]],

Cell["Here is the integral estimate.", "Text",
 CellID->530032527],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"f", "[", "abscGK", "]"}], "derivativesGK"}], ")"}], ".", 
  "weights"}]], "Input",
 CellLabel->"In[38]:=",
 CellID->542244712],

Cell[BoxData["0.4843754920980107`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[38]=",
 CellID->263204247]
}, Open  ]],

Cell["\<\
With the IMT transformation, these are the sampling points and derivatives.\
\>", "Text",
 CellID->423761297],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"abscGKIMT", "=", 
  RowBox[{
   RowBox[{
    RowBox[{"Rescale", "[", 
     RowBox[{
      RowBox[{"\[CurlyPhi]", "[", "#1", "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", 
        FractionBox["1", "16"]}], "}"}]}], "]"}], "&"}], "/@", 
   RowBox[{
    RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
     RowBox[{"5", ",", "MachinePrecision"}], "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}]}]}]], "Input",
 CellLabel->"In[39]:=",
 CellID->484576425],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "4.489421608827582`*^-56", ",", "9.37893274519999`*^-11", ",", 
   "0.00004976573191026807`", ",", "0.0022294618981518305`", ",", 
   "0.010578397210517348`", ",", "0.022992465073215146`", ",", 
   "0.03559535922918402`", ",", "0.046301442853467954`", ",", 
   "0.054327148689713806`", ",", "0.059498292542853624`", ",", 
   "0.06200068354839505`"}], "}"}]], "Output",
 ImageSize->{460, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[39]=",
 CellID->8946805]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"derivativesGKIMT", "=", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"D", "[", 
      RowBox[{
       RowBox[{"Rescale", "[", 
        RowBox[{
         RowBox[{"\[CurlyPhi]", "[", "t", "]"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", 
           FractionBox["1", "16"]}], "}"}]}], "]"}], ",", "t"}], "]"}], "/.", 
     
     RowBox[{"t", "\[Rule]", "#"}]}], "&"}], "/@", 
   RowBox[{
    RowBox[{"NIntegrate`GaussKronrodRuleData", "[", 
     RowBox[{"5", ",", "MachinePrecision"}], "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}]}]}]], "Input",
 CellLabel->"In[40]:=",
 CellID->196812001],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "7.090171677134307`*^-52", ",", "4.262075408886575`*^-8", ",", 
   "0.0032938897501190536`", ",", "0.04186574999372732`", ",", 
   "0.0815397026676444`", ",", "0.09196986029286058`", ",", 
   "0.08695293877773633`", ",", "0.07824864786352428`", ",", 
   "0.0706212118144752`", ",", "0.06549931334928846`", ",", 
   "0.06299930576459968`"}], "}"}]], "Output",
 ImageSize->{452, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[40]=",
 CellID->175573991]
}, Open  ]],

Cell["Here is the integral estimate with the IMT transformation.", "Text",
 CellID->461454413],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"f", "[", "abscGKIMT", "]"}], "derivativesGKIMT"}], ")"}], ".", 
  "weights"}]], "Input",
 CellLabel->"In[41]:=",
 CellID->97503878],

Cell[BoxData["0.500561732942949`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->49025866]
}, Open  ]],

Cell["\<\
The estimate calculated with the IMT variable transformation is much closer \
to the exact value.\
\>", "Text",
 CellID->1870738],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["x"]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     FractionBox["1", "16"]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[42]:=",
 CellID->159569001],

Cell[BoxData[
 FractionBox["1", "2"]], "Output",
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[42]=",
 CellID->1868060]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Use of double-exponential quadrature", "Subsection",
 CellTags->"UseOfDoubleExponential",
 CellID->225088061],

Cell[TextData[{
 "When adaptive strategies use the IMT variable transformation they do not \
change the integration rule on the IMT-transformed regions. In contrast to \
this you can both use a variable transformation and a different integration \
rule on the regions considered to have singularity. (This is more in the \
spirit of the",
 StyleBox[" ",
  FontSlant->"Italic"],
 "IMT rule [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "].) Exactly that happens when ",
 ButtonBox["double-exponential quadrature",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#526196975"],
 " is used\[LongDash]double-exponential quadrature uses the trapezoidal \
rule."
}], "Text",
 CellID->773413410],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can use double-exponential quadrature for singularity handling only for \
one-dimensional integration."
}], "Text",
 CellID->303661233],

Cell["\<\
Here is a numerical integration that uses double-exponential quadrature for \
singularity handling.\
\>", "MathCaption",
 CellID->420054766],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"1", "-", "x"}], "]"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
      RowBox[{
      "\"\<SingularityHandler\>\"", "\[Rule]", 
       "\"\<DoubleExponential\>\""}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[103]:=",
 CellID->297266233],

Cell[BoxData["2.0000000000000155`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[103]=",
 CellID->284521485]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
IMT versus \"DoubleExponential\" versus no singularity handling for \
one-dimensional integrals\
\>", "Subsection",
 CellTags->"IMTvsDEvsNoSH",
 CellID->23700706],

Cell[TextData[{
 "Both singularity handlers (",
 Cell[BoxData["IMT"], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 ") are applied to regions that are obtained through too many bisections (as \
specified by ",
 Cell[BoxData["\"\<SingularityDepth\>\""], "InlineFormula"],
 "). "
}], "Text",
 CellID->148021494],

Cell[TextData[{
 "The main difference between them is that ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " does not change the integration rule used to compute integral estimates on \
the region it is applied to\[LongDash]",
 Cell[BoxData["IMT"], "InlineFormula"],
 " is only a variable transformation. On the other hand, ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " uses both variable transformation and a different integration rule\
\[LongDash]the trapezoidal rule\[LongDash]to compute integral estimates on \
the region it is applied to. In other words, the singularity handler ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " delegates the integration to the double-exponential quadrature as \
described in ",
 ButtonBox["Double-Exponential Strategy",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#526196975"],
 "."
}], "Text",
 CellID->262786683],

Cell[TextData[{
 "As a consequence, a region to which the ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " singularity handler is applied to is still going to be subject to \
bisection by the adaptive integration strategy. Therefore, until the \
precision goal is reached the integrand evaluations done before the last \
bisection will be thrown away. On the other hand, a region to which the ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " singularity handler is applied will not be bisected. The trapezoidal rule \
quadrature used by ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " will compute integral estimates over the region with increasing number of \
sampling points at each step, completely reusing the integrand evaluations of \
the sampling points from the previous steps."
}], "Text",
 CellID->180087955],

Cell[TextData[{
 "So, if the integrand is \"very\" analytic (i.e. no rapid or sudden changes \
of the integrand and its derivatives wrt the integration variable) over the \
regions with end point singularity, the ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " singularity handler is going to be (much) faster than the ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " singularity handler. In the cases that the integrand is not analytic in \
the region given to the ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " singularity handler, or the double transformation of the integrand \
converges too slowly, it is better to switch to the ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " singularity handler. This is done if the option ",
 Cell[BoxData["\"\<SingularityHandler\>\""], "InlineFormula"],
 " is set to ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 "."
}], "Text",
 CellID->39516642],

Cell[TextData[{
 "Following are tables that compare the ",
 Cell[BoxData["IMT"], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 ", and ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " singularity handlers applied at different depths of bisection."
}], "Text",
 CellID->95518980],

Cell[TextData[{
 "This loads a package that defines the profiling function ",
 Cell[BoxData["NIntegrateProfile"], "InlineFormula"],
 " that gives the number of sampling points and the time needed by a \
numerical integration command."
}], "MathCaption",
 CellID->26392904],

Cell[BoxData[
 RowBox[{
  RowBox[{"Needs", "[", "\"\<Integration`NIntegrateUtilities`\>\"", "]"}], 
  ";"}]], "Input",
 CellLabel->"In[17]:=",
 CellID->1072960097],

Cell[TextData[{
 "Table for a \"very\" analytical integrand ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", 
    SqrtBox["x"]], TraditionalForm]], "InlineMath"],
 " that the ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " singularity handler easily computes."
}], "MathCaption",
 CellID->286274941],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"exact", "=", "2"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"t", "=", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
           "\"\<IntegralEstimate\>\"", ",", "\"\<Evaluations\>\"", ",", 
            "\"\<Timing\>\""}], "}"}], "/.", 
          RowBox[{"NIntegrateProfile", "[", 
           RowBox[{"NIntegrate", "[", 
            RowBox[{
             FractionBox["1", 
              SqrtBox["x"]], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
             RowBox[{"Method", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
                RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "1", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "2", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], 
               "}"}]}], ",", 
             RowBox[{"MaxRecursion", "\[Rule]", "100"}]}], "]"}], "]"}]}]}], 
        ";", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"#1", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{"t", "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "1"}], "]"}], "]"}], "-", "exact"}], "]"}], 
          ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "3", "]"}], "]"}]}], "}"}]}], ")"}], "&"}], ")"}], "/@",
     
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "Infinity"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DoubleExponential\>\"", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DoubleExponential\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "4"}], "}"}]}], "}"}]}]}], ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{"tbl", ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", 
         RowBox[{"FontFamily", "\[Rule]", "Times"}], ",", 
         RowBox[{"FontSize", "\[Rule]", "11"}]}], "]"}], "&"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
         "\"\<No singularity handling\>\"", ",", "\"\<IMT\>\"", ",", 
          "\"\<DoubleExponential\>\"", ",", "\"\<IMT\>\"", ",", 
          "\"\<DoubleEponential\>\"", ",", "\"\<Automatic\>\""}], "}"}], ",", 
        
        RowBox[{"{", 
         RowBox[{"\"\<SingularityDepth\>\"", ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Difference from\>\"", ",", "\"\<the exact integral\>\""}], 
            "}"}], "]"}], ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Number of function\>\"", ",", "\"\<evaluations\>\""}], "}"}],
            "]"}], ",", "\"\<Time (s)\>\""}], "}"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"-", "1"}], "}"}]}], "]"}]}]}], "]"}]}], "Input",
 CellLabel->"In[34]:=",
 CellID->647347635],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{463, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[36]//TableForm=",
 CellID->436779486]
}, Open  ]],

Cell[TextData[{
 "Table for an integrand, ",
 Cell[BoxData[
  FormBox[
   FractionBox["70", 
    RowBox[{
     RowBox[{
      SuperscriptBox["10", "4"], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"x", "-", 
         FractionBox["1", "32"]}], ")"}], "2"]}], "+", 
     FractionBox["1", "16"]}]], TraditionalForm]], "InlineMath"],
 ", that does not have a singularity and has a nearly discontinuous \
derivative. (I.e. it is not \"very\" analytical.) The ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " singularity handler starts with ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " and then switches to ",
 Cell[BoxData["IMT"], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->3057156],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   FractionBox["70", 
    RowBox[{
     RowBox[{
      SuperscriptBox["10", "4"], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"x", "-", 
         FractionBox["1", "32"]}], ")"}], "2"]}], "+", 
     FractionBox["1", "16"]}]]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"exact", "=", 
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"t", "=", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
           "\"\<IntegralEstimate\>\"", ",", "\"\<Evaluations\>\"", ",", 
            "\"\<Timing\>\""}], "}"}], "/.", 
          RowBox[{"NIntegrateProfile", "[", 
           RowBox[{"NIntegrate", "[", 
            RowBox[{
             RowBox[{"f", "[", "x", "]"}], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
             RowBox[{"Method", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
                RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "1", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "2", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], 
               "}"}]}], ",", 
             RowBox[{"MaxRecursion", "\[Rule]", "100"}], ",", 
             RowBox[{"PrecisionGoal", "\[Rule]", "8"}]}], "]"}], "]"}]}]}], 
        ";", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"#1", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{"t", "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "1"}], "]"}], "]"}], "-", "exact"}], "]"}], 
          ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "3", "]"}], "]"}]}], "}"}]}], ")"}], "&"}], ")"}], "/@",
     
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "Infinity"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DoubleExponential\>\"", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DoubleExponential\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "4"}], "}"}]}], "}"}]}]}], ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{"tbl", ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", 
         RowBox[{"FontFamily", "\[Rule]", "Times"}], ",", 
         RowBox[{"FontSize", "\[Rule]", "11"}]}], "]"}], "&"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
         "\"\<No singularity handling\>\"", ",", "\"\<IMT\>\"", ",", 
          "\"\<DoubleExponential\>\"", ",", "\"\<IMT\>\"", ",", 
          "\"\<DoubleEponential\>\"", ",", "\"\<Automatic\>\""}], "}"}], ",", 
        
        RowBox[{"{", 
         RowBox[{"\"\<SingularityDepth\>\"", ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Difference\>\"", ",", "\"\<from the exact integral\>\""}], 
            "}"}], "]"}], ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Number of function\>\"", ",", "\"\<evaluations\>\""}], "}"}],
            "]"}], ",", "\"\<Time (s)\>\""}], "}"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"-", "1"}], "}"}]}], "]"}]}]}], "]"}]}], "Input",
 CellLabel->"In[37]:=",
 CellID->133788890],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{480, 137},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[40]//TableForm=",
 CellID->111735794]
}, Open  ]],

Cell[TextData[{
 "A table for an integrand, ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"x", "+", 
     FractionBox["1", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"Log", "[", "x", "]"}]}]]}], 
    RowBox[{"x", " ", 
     RowBox[{"Log", "[", "x", "]"}]}]], TraditionalForm]], "InlineMath"],
 ", for which the ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " singularity handler starts with ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " and then switches to ",
 Cell[BoxData["IMT"], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->615909903],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   FractionBox[
    RowBox[{"x", "+", 
     FractionBox["1", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"Log", "[", "x", "]"}]}]]}], 
    RowBox[{"x", " ", 
     RowBox[{"Log", "[", "x", "]"}]}]]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"exact", "=", 
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"t", "=", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
           "\"\<IntegralEstimate\>\"", ",", "\"\<Evaluations\>\"", ",", 
            "\"\<Timing\>\""}], "}"}], "/.", 
          RowBox[{"NIntegrateProfile", "[", 
           RowBox[{"NIntegrate", "[", 
            RowBox[{
             RowBox[{"f", "[", "x", "]"}], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
             RowBox[{"Method", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
                RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "1", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "2", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], 
               "}"}]}], ",", 
             RowBox[{"MaxRecursion", "\[Rule]", "3000"}], ",", 
             RowBox[{"PrecisionGoal", "\[Rule]", "6"}]}], "]"}], "]"}]}]}], 
        ";", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"#1", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{"t", "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "1"}], "]"}], "]"}], "-", "exact"}], "]"}], 
          ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "3", "]"}], "]"}]}], "}"}]}], ")"}], "&"}], ")"}], "/@",
     
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "Infinity"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DoubleExponential\>\"", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DoubleExponential\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "4"}], "}"}]}], "}"}]}]}], ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{"tbl", ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", 
         RowBox[{"FontFamily", "\[Rule]", "Times"}], ",", 
         RowBox[{"FontSize", "\[Rule]", "11"}]}], "]"}], "&"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
         "\"\<No singularity handling\>\"", ",", "\"\<IMT\>\"", ",", 
          "\"\<DoubleExponential\>\"", ",", "\"\<IMT\>\"", ",", 
          "\"\<DoubleEponential\>\"", ",", "\"\<Automatic\>\""}], "}"}], ",", 
        
        RowBox[{"{", 
         RowBox[{"\"\<SingularityDepth\>\"", ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Difference from\>\"", ",", " ", 
             "\"\<the exact integral\>\""}], "}"}], "]"}], ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Number of function\>\"", ",", "\"\<evaluations\>\""}], "}"}],
            "]"}], ",", "\"\<Time (s)\>\""}], "}"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"-", "1"}], "}"}]}], "]"}]}]}], "]"}]}], "Input",
 CellLabel->"In[41]:=",
 CellID->426823284],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{463, 133},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[44]//TableForm=",
 CellID->520233779]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["IMT multidimensional singularity handling", "Subsection",
 CellTags->"IMTforMD",
 CellID->159907828],

Cell[TextData[{
 "When used for multidimensional integrals, the IMT singularity handler \
speeds up the integration process only when the singularity is along one of \
the axes. When the singularity is at a corner of the integration region, \
using IMT is counterproductive. The function ",
 Cell[BoxData["NIntegrateProfile"], "InlineFormula"],
 " defined earlier is used in the following examples."
}], "Text",
 CellID->433332065],

Cell[TextData[{
 "The number of integrand evaluations and timings for an integrand that has a \
singularity only along the ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " axis. The default, automatic, singularity handler chooses to apply IMT to \
regions obtained after the default, four, bisections. "
}], "MathCaption",
 CellID->367753712],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrateProfile", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", 
      RowBox[{"Sqrt", "[", "x", "]"}]], "+", "y"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[19]:=",
 CellID->51478290],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\<\"IntegralEstimate\"\>", "\[Rule]", 
    InterpretationBox[
     StyleBox["2.500000004270092",
      ShowStringCharacters->True,
      NumberMarks->True],
     InputForm[2.500000004270092],
     AutoDelete->True,
     Editable->True]}], ",", 
   RowBox[{"\<\"Evaluations\"\>", "\[Rule]", "442"}], ",", 
   RowBox[{"\<\"Timing\"\>", "\[Rule]", "0.025000000000000064`"}]}], 
  "}"}]], "Output",
 ImageSize->{495, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->536214380]
}, Open  ]],

Cell[TextData[{
 "The number of integrand evaluations and timings for an integrand that has a \
singularity only along the ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " axis with no singularity handler application."
}], "MathCaption",
 CellID->367512733],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrateProfile", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", 
      RowBox[{"Sqrt", "[", "x", "]"}]], "+", "y"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], "}"}]}], 
    ",", 
    RowBox[{"MaxRecursion", "\[Rule]", "30"}]}], "]"}]}]], "Input",
 CellLabel->"In[20]:=",
 CellID->141740299],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\<\"IntegralEstimate\"\>", "\[Rule]", 
    InterpretationBox[
     StyleBox["2.4999994380778543",
      ShowStringCharacters->True,
      NumberMarks->True],
     InputForm[2.4999994380778543`],
     AutoDelete->True,
     Editable->True]}], ",", 
   RowBox[{"\<\"Evaluations\"\>", "\[Rule]", "1445"}], ",", 
   RowBox[{"\<\"Timing\"\>", "\[Rule]", "0.023099999999999832`"}]}], 
  "}"}]], "Output",
 ImageSize->{516, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->212615798]
}, Open  ]],

Cell[TextData[{
 "The number of integrand evaluations and timings for an integrand that has a \
singularity at a corner of the integration region. The default, automatic, \
singularity handler chooses to apply the singularity handler ",
 StyleBox[ButtonBox["DuffyCoordinates",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#738848244"], 
  "InlineCode"],
 " to regions obtained after the default, four, bisections. "
}], "MathCaption",
 CellID->94020458],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrateProfile", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{"Sqrt", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], "]"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[21]:=",
 CellID->456292875],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\<\"IntegralEstimate\"\>", "\[Rule]", 
    InterpretationBox[
     StyleBox["1.7627471522176814",
      ShowStringCharacters->True,
      NumberMarks->True],
     InputForm[1.7627471522176814`],
     AutoDelete->True,
     Editable->True]}], ",", 
   RowBox[{"\<\"Evaluations\"\>", "\[Rule]", "2006"}], ",", 
   RowBox[{"\<\"Timing\"\>", "\[Rule]", "0.037999999999999985`"}]}], 
  "}"}]], "Output",
 ImageSize->{509, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[21]=",
 CellID->179585300]
}, Open  ]],

Cell["\<\
The number of integrand evaluations and timings for an integrand that has a \
singularity at a corner of the integration region. IMT is applied to regions \
obtained after the default, four, bisections. \
\>", "MathCaption",
 CellID->157581889],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrateProfile", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{"Sqrt", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], "]"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "\"\<IMT\>\""}]}], 
      "}"}]}], ",", 
    RowBox[{"MaxRecursion", "\[Rule]", "30"}]}], "]"}]}]], "Input",
 CellLabel->"In[22]:=",
 CellID->669464312],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[22]:=",
 CellID->354550602],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[22]:=",
 CellID->44000563],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[22]:=",
 CellID->550517120],

Cell[BoxData[
 RowBox[{
  RowBox[{"General", "::", "\<\"stop\"\>"}], ":", 
  " ", "\<\"Further output of \\!\\(NIntegrate :: \\\"slwcon\\\"\\) will be \
suppressed during this calculation. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\
\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/General/stop\\\", ButtonNote -> \
\\\"General::stop\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[22]:=",
 CellID->153358539],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\<\"IntegralEstimate\"\>", "\[Rule]", 
    InterpretationBox[
     StyleBox["1.762747132592934",
      ShowStringCharacters->True,
      NumberMarks->True],
     InputForm[1.762747132592934],
     AutoDelete->True,
     Editable->True]}], ",", 
   RowBox[{"\<\"Evaluations\"\>", "\[Rule]", "7004"}], ",", 
   RowBox[{"\<\"Timing\"\>", "\[Rule]", "0.09410000000000017`"}]}], 
  "}"}]], "Output",
 ImageSize->{509, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[22]=",
 CellID->21172427]
}, Open  ]],

Cell["\<\
The number of integrand evaluations and timings for an integrand that has a \
singularity at a corner of the integration region with no singularity handler \
application.\
\>", "MathCaption",
 CellID->459722911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrateProfile", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{"Sqrt", "[", 
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], "]"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], "}"}]}], 
    ",", 
    RowBox[{"MaxRecursion", "\[Rule]", "30"}]}], "]"}]}]], "Input",
 CellLabel->"In[23]:=",
 CellID->46821559],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"\<\"IntegralEstimate\"\>", "\[Rule]", 
    InterpretationBox[
     StyleBox["1.7627469943973395",
      ShowStringCharacters->True,
      NumberMarks->True],
     InputForm[1.7627469943973395`],
     AutoDelete->True,
     Editable->True]}], ",", 
   RowBox[{"\<\"Evaluations\"\>", "\[Rule]", "3791"}], ",", 
   RowBox[{"\<\"Timing\"\>", "\[Rule]", "0.04510000000000007`"}]}], 
  "}"}]], "Output",
 ImageSize->{516, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->701100772]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Duffy's coordinates for multidimensional singularity handling\
\>", "Subsection",
 CellTags->"DuffyCoordinatesTheory",
 CellID->738848244],

Cell["\<\
Duffy's coordinates is a technique that transforms an integrand over a \
square, cube, or hypercube with a singular point in one of the corners into \
an integrand with a singularity over a line, which might be easier to \
integrate.\
\>", "Text",
 CellID->265514329],

Cell["The following integration uses Duffy's coordinates.", "MathCaption",
 CellID->186812155],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{
       "\"\<SingularityHandler\>\"", "\[Rule]", 
        "\"\<DuffyCoordinates\>\""}]}], "}"}]}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[63]:=",
 CellID->526011462],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.017996999999998493`", ",", "1.762747108980918`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[63]=",
 CellID->501272045]
}, Open  ]],

Cell["The following integration does not use Duffy's coordinates.", \
"MathCaption",
 CellID->96244487],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"\"\<SingularityHandler\>\"", "->", "None"}]}], "}"}]}], ",", 
    RowBox[{"MaxRecursion", "->", "20"}]}], "]"}], "//", "Timing"}]], "Input",\

 CellLabel->"In[62]:=",
 CellID->3694344],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.03899399999999695`", ",", "1.7627469943973395`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[62]=",
 CellID->89162723]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s strategies ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " apply the Duffy's coordinates technique only at the corners of the \
integration region."
}], "Text",
 CellID->75767650],

Cell[TextData[{
 "When the singularity of a multidimensional integral occurs at a point, the \
coupling of the variables will make the singularity variable transformations \
used in one-dimensional integration counterproductive. A variable \
transformation that has a geometrical nature, proposed by Duffy in [",
 ButtonBox["Duffy82",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#138937541"],
 "], makes a change of variables that replaces a point singularity at a \
corner of the integration region with a \"softer\" one on a plane. "
}], "Text",
 CellID->314606834],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " is the dimension of integration and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"r", "=", 
    RowBox[{
     SuperscriptBox[
      SubscriptBox["x", "1"], "2"], "+", 
     SuperscriptBox[
      SubscriptBox["x", "2"], "2"], "+", "\[Ellipsis]", "+", 
     SuperscriptBox[
      SubscriptBox["x", "d"], "2"]}]}], TraditionalForm]], "InlineMath"],
 ", then Duffy's coordinates is a suitable technique for singularities of the \
following type (see again [",
 ButtonBox["Duffy82",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#138937541"],
 "]):\n1. ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["r", "\[Alpha]"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["r", "\[Alpha]"], " ", "ln", " ", "r"}], TraditionalForm]],
   "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Alpha]", ">", 
    RowBox[{"-", "d"}]}], TraditionalForm]], "InlineMath"],
 " ;\n2. ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     SubscriptBox["x", "1"], 
     SubscriptBox["\[Alpha]", "1"]], 
    SuperscriptBox[
     SubscriptBox["x", "2"], 
     SubscriptBox["\[Alpha]", "2"]], 
    SuperscriptBox[
     SubscriptBox["\[Ellipsis]x", "d"], 
     SubscriptBox["\[Alpha]", "d"]], 
    SuperscriptBox["r", "\[Beta]"]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["\[Alpha]", "i"], ">", 
     RowBox[{"-", "1"}]}], ",", " ", 
    RowBox[{"i", "\[Element]", 
     RowBox[{"[", 
      RowBox[{"1", ",", "d"}], "]"}]}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"\[Sum]", 
      SubscriptBox["\[Alpha]", "i"]}], "+", "\[Beta]"}], ">", 
    RowBox[{"-", "d"}]}], TraditionalForm]], "InlineMath"],
 " ;\n3. ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{
      RowBox[{
       SubscriptBox["c", "1"], 
       SuperscriptBox[
        SubscriptBox["x", "1"], "\[Beta]"]}], "+", 
      RowBox[{
       SubscriptBox["c", "2"], 
       SuperscriptBox[
        SubscriptBox["x", "2"], "\[Beta]"]}], "+", "\[Ellipsis]", "+", 
      RowBox[{
       SubscriptBox["c", "d"], 
       SuperscriptBox[
        SubscriptBox["x", "d"], "\[Beta]"]}]}], ")"}], "\[Alpha]"], 
   TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Beta]", ">", "0"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[Alpha]", " ", "\[Beta]"}], ">", 
    RowBox[{"-", "d"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["c", "i"], ">", "0"}], ",", " ", 
    RowBox[{"i", "\[Element]", 
     RowBox[{"[", 
      RowBox[{"1", ",", "d"}], "]"}]}]}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellTags->"DuffyCoordsSingFuncs",
 CellID->574465939],

Cell["For example, consider the integral", "Text",
 CellID->246511656],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubsuperscriptBox["\[Integral]", "0", "1"], 
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "x"], 
    RowBox[{
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        RowBox[{"4", 
         SuperscriptBox["x", "2"]}], "+", 
        SuperscriptBox["y", "2"]}]]], 
     RowBox[{"\[DifferentialD]", "x"}], 
     RowBox[{
      RowBox[{"\[DifferentialD]", "y"}], "."}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->223736977],

Cell[TextData[{
 "If the integration region ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
    RowBox[{"(", 
     RowBox[{"0", ",", "x"}], "]"}]}], TraditionalForm]], "InlineMath"],
 " is changed to ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
    RowBox[{"(", 
     RowBox[{"0", ",", "1"}], "]"}]}], TraditionalForm]], "InlineMath"],
 " with the rule ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[Rule]", " ", 
    RowBox[{"x", " ", "y"}]}], TraditionalForm]], "InlineMath"],
 " the Jacobian of which is ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 ", the integral becomes"
}], "Text",
 CellID->518400971],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "x"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", "\[DoubleLeftRightArrow]",
    
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["x", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"x", " ", "y"}], ")"}], "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", "\[DoubleLeftRightArrow]",
    
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FormBox[
       FractionBox["1", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["y", "2"], "+", "4"}]]],
       TraditionalForm], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{
       RowBox[{"\[DifferentialD]", "y"}], "."}]}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"DuffyCoordinates1",
 CellID->178372338],

Cell["The last integral has no singularities at all!", "Text",
 CellID->391365016],

Cell["Now consider the integral", "Text",
 CellID->706516988],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"DuffyCoordinates2",
 CellID->133554823],

Cell["which is equivalent to the sum", "Text",
 CellID->369539985],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "x"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", "+", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "x", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{
       RowBox[{"\[DifferentialD]", "y"}], "."}]}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->75397808],

Cell[TextData[{
 "The first integral of that sum is transformed as in (",
 
 CounterBox["NumberedEquation", "DuffyCoordinates1"],
 "); for the second one, though, the change of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
    RowBox[{"(", 
     RowBox[{"1", ",", "x"}], "]"}]}], TraditionalForm]], "InlineMath"],
 " into ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"0", ",", "1"}], "]"}], "\[Times]", 
    RowBox[{"(", 
     RowBox[{"0", ",", "1"}], "]"}]}], TraditionalForm]], "InlineMath"],
 " by ",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[Rule]", 
    RowBox[{"x", "+", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"1", "-", "x"}], ")"}], "y"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " has the Jacobian ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "-", "x"}], TraditionalForm]], "InlineMath"],
 ", which will not bring the desired cancellation of terms. Fortunately, a \
change of the order of integration:"
}], "Text",
 CellID->288580156],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "x", "1"], 
      RowBox[{
       FractionBox["1", 
        SqrtBox[
         RowBox[{
          RowBox[{"4", 
           SuperscriptBox["x", "2"]}], "+", 
          SuperscriptBox["y", "2"]}]]], 
       RowBox[{"\[DifferentialD]", "x"}], 
       RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", 
    "\[DoubleLeftRightArrow]", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "y"], 
      RowBox[{
       FractionBox["1", 
        SqrtBox[
         RowBox[{
          RowBox[{"4", 
           SuperscriptBox["x", "2"]}], "+", 
          SuperscriptBox["y", "2"]}]]], 
       RowBox[{"\[DifferentialD]", "y"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->25730526],

Cell[TextData[{
 "makes the second integral amenable for the transformation in (",
 
 CounterBox["NumberedEquation", "DuffyCoordinates1"],
 "):",
 " "
}], "Text",
 CellID->391236153],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "y"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], " ", "\[DoubleLeftRightArrow]",
    
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "x"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["y", "2"]}], "+", 
         SuperscriptBox["x", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", "\[DoubleLeftRightArrow]",
    
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["x", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", " ", "y"}], ")"}], "2"]}], "+", 
         SuperscriptBox["x", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", "\[DoubleLeftRightArrow]",
    
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{"1", "+", 
         RowBox[{"4", 
          SuperscriptBox["y", "2"]}]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{
       RowBox[{"\[DifferentialD]", "y"}], "."}]}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"DuffyCoordinates3",
 CellID->372923878],

Cell["\<\
(In the second integral in the equation (3) the variables were permuted, \
which is not necessary to prove the mathematical equivalence, but it pays off \
when computing the integrals.) \
\>", "Text",
 CellID->650094527],

Cell[TextData[{
 "So the integral (",
 
 CounterBox["NumberedEquation", "DuffyCoordinates2"],
 ") can be rewritten as an integral with no singularities:"
}], "Text",
 CellID->16880680],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", "\[DoubleLeftRightArrow]",
    
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      FormBox[
       FractionBox["1", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["y", "2"], "+", "4"}]]],
       TraditionalForm]}]}], "+", 
    RowBox[{
     FractionBox["1", 
      SqrtBox[
       RowBox[{"1", "+", 
        RowBox[{"4", 
         SuperscriptBox["y", "2"]}]}]]], 
     RowBox[{"\[DifferentialD]", "x"}], " ", 
     RowBox[{
      RowBox[{"\[DifferentialD]", "y"}], "."}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->558855580],

Cell[TextData[{
 "If the integration variables were not permuted in (",
 
 CounterBox["NumberedEquation", "DuffyCoordinates3"],
 "), the integral (",
 
 CounterBox["NumberedEquation", "DuffyCoordinates2"],
 ") is going to be rewritten as"
}], "Text",
 CellID->499989446],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         RowBox[{"4", 
          SuperscriptBox["x", "2"]}], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], " ", "\[DoubleLeftRightArrow]",
    
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      FormBox[
       FractionBox["1", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["y", "2"], "+", "4"}]]],
       TraditionalForm]}]}], "+", 
    RowBox[{
     FractionBox["1", 
      SqrtBox[
       RowBox[{"1", "+", 
        RowBox[{"4", 
         SuperscriptBox["x", "2"]}]}]]], 
     RowBox[{"\[DifferentialD]", "x"}], " ", 
     RowBox[{
      RowBox[{"\[DifferentialD]", "y"}], "."}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->203916911],

Cell["\<\
That is a more complicated integral, as its integrand is not simple along \
both axes. Subsequently it is harder to compute than the former one.\
\>", "Text",
 CellID->72735120],

Cell["\<\
Here is the number of sampling points for the simpler integral.\
\>", "MathCaption",
 CellID->625932685],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Reap", "[", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{
       FractionBox["1", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["y", "2"], "+", "4"}]]], "+", 
       FractionBox["1", 
        SqrtBox[
         RowBox[{"1", "+", 
          RowBox[{"4", " ", 
           SuperscriptBox["y", "2"]}]}]]]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
      RowBox[{"PrecisionGoal", "\[Rule]", "8"}], ",", 
      RowBox[{"EvaluationMonitor", ":>", 
       RowBox[{"Sow", "[", 
        RowBox[{"{", 
         RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"2", ",", "1"}], "]"}], "]"}], "//", "Length"}]], "Input",
 CellLabel->"In[58]:=",
 CellID->722556286],

Cell[BoxData["187"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[58]=",
 CellID->105346464]
}, Open  ]],

Cell["\<\
Here is the number of sampling points for the more complicated integral.\
\>", "MathCaption",
 CellID->73417775],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Reap", "[", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{
       FractionBox["1", 
        SqrtBox[
         RowBox[{
          SuperscriptBox["y", "2"], "+", "4"}]]], "+", 
       FractionBox["1", 
        SqrtBox[
         RowBox[{"1", "+", 
          RowBox[{"4", " ", 
           SuperscriptBox["x", "2"]}]}]]]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
      RowBox[{"PrecisionGoal", "\[Rule]", "8"}], ",", 
      RowBox[{"EvaluationMonitor", ":>", 
       RowBox[{"Sow", "[", 
        RowBox[{"{", 
         RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"2", ",", "1"}], "]"}], "]"}], "//", "Length"}]], "Input",
 CellLabel->"In[59]:=",
 CellID->36525702],

Cell[BoxData["323"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[59]=",
 CellID->236177499]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " uses a generalization to arbitrary dimension of the technique in the \
example above. (In [",
 ButtonBox["Duffy82",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#138937541"],
 "] third dimension is described only.) An ",
 ButtonBox["example implementation together with the generalization \
description",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#354695682"],
 " are given below."
}], "Text",
 CellID->308288215],

Cell[TextData[{
 "Here is a table that compares the different singularity handlings for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"]}]]], 
      RowBox[{"\[DifferentialD]", "x"}], 
      RowBox[{"\[DifferentialD]", "y"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". (The profiling function ",
 Cell[BoxData["NIntegrateProfile"], "InlineFormula"],
 ButtonBox[" defined earlier",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#515752656"],
 " is used.)"
}], "MathCaption",
 CellID->280282289],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"exact", "=", 
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"1", "/", 
      RowBox[{"Sqrt", "[", 
       RowBox[{
        RowBox[{"x", "^", "2"}], " ", "+", " ", 
        RowBox[{"y", "^", "2"}]}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]}], ";", 
  RowBox[{"tbl", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"t", "=", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
           "\"\<IntegralEstimate\>\"", ",", "\"\<Evaluations\>\"", ",", 
            "\"\<Timing\>\""}], "}"}], "/.", 
          RowBox[{"NIntegrateProfile", "[", 
           RowBox[{"NIntegrate", "[", 
            RowBox[{
             RowBox[{"1", "/", 
              RowBox[{"Sqrt", "[", 
               RowBox[{
                RowBox[{"x", "^", "2"}], "+", 
                RowBox[{"y", "^", "2"}]}], "]"}]}], ",", 
             RowBox[{"{", 
              RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
             RowBox[{"Method", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
                RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "1", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", 
                 RowBox[{"#1", "[", 
                  RowBox[{"[", "2", "]"}], "]"}]}], ",", 
                RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], 
               "}"}]}], ",", 
             RowBox[{"MaxRecursion", "\[Rule]", "12"}]}], "]"}], "]"}]}]}], 
        ";", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"#1", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"Abs", "[", 
           RowBox[{
            RowBox[{"t", "[", 
             RowBox[{"[", 
              RowBox[{"1", ",", "1"}], "]"}], "]"}], "-", "exact"}], "]"}], 
          ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ",", 
          RowBox[{"t", "[", 
           RowBox[{"[", "3", "]"}], "]"}]}], "}"}]}], ")"}], "&"}], ")"}], "/@",
     
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"None", ",", "Infinity"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<IMT\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DuffyCoordinates\>\"", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"\"\<DuffyCoordinates\>\"", ",", "1"}], "}"}]}], "}"}]}]}], 
  ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{"tbl", ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", 
         RowBox[{"FontFamily", "\[Rule]", "Times"}], ",", 
         RowBox[{"FontSize", "\[Rule]", "11"}]}], "]"}], "&"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
         "\"\<No singularity handling\>\"", ",", "\"\<IMT\>\"", ",", 
          "\"\<IMT\>\"", ",", "\"\<DuffyCoordinates\>\"", ",", 
          "\"\<DuffyCoordinates\>\""}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"\"\<SingularityDepth\>\"", ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Difference\>\"", ",", "\"\<from the\>\"", ",", 
             "\"\<nexact integral\>\""}], "}"}], "]"}], ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<Number of\>\"", ",", "\"\<function\>\"", ",", 
             "\"\<evaluations\>\""}], "}"}], "]"}], ",", "\"\<Time (s)\>\""}],
          "}"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"-", "1"}], "}"}]}], "]"}]}]}], "]"}]}], "Input",
 CellLabel->"In[74]:=",
 CellID->13690822],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{418, 133},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[75]=",
 CellID->7836532]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Duffy's coordinates strategy", "Subsection",
 CellTags->"DuffyCoordinates",
 CellID->66551248],

Cell[TextData[{
 "When Duffy's coordinates are applicable, a numerical integration result is \
obtained faster if Duffy's coordinate change is made before the actual \
integration begins. Making the transformation beforehand, though, requires \
knowledge at which corner(s) of the integration region the singularities \
occur. ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " strategy facilitates such pre-integration transformation. "
}], "Text",
 CellID->490944724],

Cell["\<\
Here is an example with an integrand that has singularities at two different \
corners of its integration region.\
\>", "MathCaption",
 CellID->403157214],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}]]], "+", 
    FractionBox["1", 
     SqrtBox[
      RowBox[{"x", "+", 
       RowBox[{"(", 
        RowBox[{"1", "-", "y"}], ")"}]}]]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<DuffyCoordinates\>\"", ",", 
      RowBox[{"\"\<Corners\>\"", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[84]:=",
 CellID->93124050],

Cell[BoxData["2.867316788010502`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[84]=",
 CellID->79124675]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    RowBox[{"{", 
     RowBox[{
      ButtonBox["\"\<GlobalAdaptive\>\"",
       BaseStyle->"Link",
       ButtonData->
        "paclet:tutorial/NIntegrateIntegrationStrategies#137878679"], ",", 
      RowBox[{"\"\<SingularityDepth\>\"", "->", "\[Infinity]"}]}], "}"}], 
    Cell["\<\
the strategy with which the integration will be made after applying Duffy's \
coordinates transformation\
\>", "TableText"]},
   {"\"\<Corners\>\"", 
    ButtonBox["All",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/All"], Cell["\<\
a vector or a list of vectors that specify the corner(s) to apply the Duffy's \
coordinates tranformation; the elements of the vectors are either 0 or 1; \
each vector length equals the dimension of the integral\
\>", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
 GridBoxItemSize->{"Columns" -> {
     Scaled[0.2], 
     Scaled[0.35], {
      Scaled[0.45]}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
   "RowsIndexed" -> {}}},
 CellID->347158452],

Cell[TextData[{
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->60449919],

Cell[TextData[{
 "The first thing ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " does is to rescale the integral into one that is over the unit hypercube \
(or square, or cube). If only one corner is specified ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " applies Duffy's coordinates transformation as described ",
 ButtonBox["earlier",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#738848244"],
 ". If more than one corner is specified, the unit hypercube of the previous \
step is partitioned into disjoint cubes with side length of one-half. \
Consider the integrals over these disjoint cubes. Duffy's coordinates \
transformation is applied to the ones that have a vertex that is specified to \
be singular. The rest are transformed into integrals over the unit cube. \
Since all integrals at this point have an integration region that is the unit \
cube, they are summated, and that sum is given to ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with a ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option that is the same as the one given to ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 "."
}], "Text",
 CellID->1615505],

Cell[TextData[{
 "The actual integrand used by ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " can be obtained through ",
 Cell[BoxData["NIntegrate`DuffyCoordinatesIntegrand"], "InlineFormula"],
 " which has the same arguments as ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 StyleBox[".", "InlineCode"]
}], "Text",
 CellID->156359738],

Cell[TextData[{
 "Here is an example for the ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " integrand of a three-dimensional function that is singular at one of the \
corners of the integration region."
}], "MathCaption",
 CellID->368910084],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate`DuffyCoordinatesIntegrand", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "3"], "+", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"1", "-", "y"}], ")"}], "3"], "+", 
       SuperscriptBox["z", "3"]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"z", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<DuffyCoordinates\>\"", ",", 
       RowBox[{"\"\<Corners\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1", ",", "0"}], "}"}]}]}], "}"}]}]}], "]"}], "//",
   "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"Simplify", "[", 
    RowBox[{"#", ",", 
     RowBox[{"Assumptions", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"0", "\[LessEqual]", "x", "\[LessEqual]", "1"}], ",", 
        RowBox[{"0", "\[LessEqual]", "y", "\[LessEqual]", " ", "1"}], ",", 
        RowBox[{"0", "\[LessEqual]", "z", "\[LessEqual]", "1"}]}], "}"}]}]}], 
    "]"}], "&"}]}]], "Input",
 CellLabel->"In[78]:=",
 CellID->11051191],

Cell[BoxData[
 RowBox[{"3", " ", 
  SqrtBox[
   FractionBox["x", 
    RowBox[{"1", "+", 
     SuperscriptBox["y", "3"], "+", 
     SuperscriptBox["z", "3"]}]]]}]], "Output",
 ImageSize->{96, 39},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[78]=",
 CellID->796799493]
}, Open  ]],

Cell[TextData[{
 "Here is an example for the ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " integrand for a two-dimensional function that is singular at two of the \
corners of the integration region."
}], "MathCaption",
 CellID->466981564],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate`DuffyCoordinatesIntegrand", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}]]], 
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"1", "-", "y"}], ")"}], "2"]}]]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<DuffyCoordinates\>\"", ",", 
       RowBox[{"\"\<Corners\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0", ",", "0"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}]}], "}"}]}]}], "}"}]}]}], "]"}], "//",
   "\[IndentingNewLine]", 
  RowBox[{
   RowBox[{"Simplify", "[", 
    RowBox[{"#", ",", 
     RowBox[{"Assumptions", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"0", "\[LessEqual]", "x", "\[LessEqual]", "1"}], ",", 
        RowBox[{"0", "\[LessEqual]", "y", "\[LessEqual]", " ", "1"}]}], 
       "}"}]}]}], "]"}], "&"}]}]], "Input",
 CellLabel->"In[79]:=",
 CellID->391869795],

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   SqrtBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "+", 
       RowBox[{"2", " ", "x"}], "+", 
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{"5", "+", 
       RowBox[{"2", " ", "x"}], "+", 
       SuperscriptBox["x", "2"], "-", 
       RowBox[{"4", " ", "y"}], "+", 
       SuperscriptBox["y", "2"]}], ")"}]}]]], "+", 
  FractionBox["1", 
   SqrtBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"2", "+", 
       RowBox[{"2", " ", "x"}], "+", 
       SuperscriptBox["x", "2"], "-", 
       RowBox[{"2", " ", "y"}], "+", 
       SuperscriptBox["y", "2"]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{"2", "+", 
       RowBox[{"2", " ", "x"}], "+", 
       SuperscriptBox["x", "2"], "+", 
       RowBox[{"2", " ", "y"}], "+", 
       SuperscriptBox["y", "2"]}], ")"}]}]]], "+", 
  FractionBox["2", 
   SqrtBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "+", 
       SuperscriptBox["y", "2"]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{"4", "-", 
       RowBox[{"4", " ", "x"}], "+", 
       RowBox[{
        SuperscriptBox["x", "2"], " ", 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          SuperscriptBox["y", "2"]}], ")"}]}]}], ")"}]}]]], "+", 
  FractionBox["2", 
   SqrtBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1", "+", 
       SuperscriptBox["y", "2"]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{"4", "-", 
       RowBox[{"4", " ", "x", " ", "y"}], "+", 
       RowBox[{
        SuperscriptBox["x", "2"], " ", 
        RowBox[{"(", 
         RowBox[{"1", "+", 
          SuperscriptBox["y", "2"]}], ")"}]}]}], ")"}]}]]]}]], "Output",
 ImageSize->{571, 83},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[79]=",
 CellID->18705708]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " might considerably improve speed for ",
 ButtonBox["the types of integrands",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#574465939"],
 " described in ",
 ButtonBox["Duffy's coordinates for multidimensional singularity handling",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#738848244"],
 "."
}], "Text",
 CellID->225715773],

Cell[TextData[{
 "Integration with ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 StyleBox[".", "Text"]
}], "MathCaption",
 CellID->465453117],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"], "+", 
        SuperscriptBox["z", "2"]}]]], "+", 
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"1", "-", "z"}], ")"}], "2"]}]]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"z", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<DuffyCoordinates\>\"", ",", 
       RowBox[{"\"\<Corners\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "0", ",", "1"}], "}"}]}], "}"}]}]}], "}"}]}]}], 
   "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[80]:=",
 CellID->214132493],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.022996999999994973`", ",", "2.3800769668259725`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[80]=",
 CellID->348853777]
}, Open  ]],

Cell[TextData[{
 "Integration with the default ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " options settings",
 StyleBox[" which is much slower than the previous one.", "Text"]
}], "MathCaption",
 CellID->10214205],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"], "+", 
        SuperscriptBox["z", "2"]}]]], "+", 
     FractionBox["1", 
      SqrtBox[
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"1", "-", "z"}], ")"}], "2"]}]]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"z", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[81]:=",
 CellID->456942207],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.25295999999999974`", ",", "2.3800773357514577`"}], 
  "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[81]=",
 CellID->684072362]
}, Open  ]],

Cell[TextData[{
 "Here is another example of a speedup by ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->110169364],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{"x", "+", 
       RowBox[{"Sin", "[", 
        RowBox[{"1", "-", "y"}], "]"}]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<DuffyCoordinates\>\"", ",", 
       RowBox[{"\"\<Corners\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}]}]}], "}"}]}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[82]:=",
 CellID->234320427],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.01099900000000098`", ",", "1.1214166445027849`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[82]=",
 CellID->132544202]
}, Open  ]],

Cell[TextData[{
 "Integration with the default ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " options settings",
 StyleBox[" which is much slower than the previous one.", "Text"]
}], "MathCaption",
 CellID->254140133],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{"x", "+", 
       RowBox[{"Sin", "[", 
        RowBox[{"1", "-", "y"}], "]"}]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[83]:=",
 CellID->269624814],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.03599400000000671`", ",", "1.121416641992694`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[83]=",
 CellID->37135782]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Duffy's coordinates generalization and example implementation\
\>", "Subsection",
 CellTags->"DuffyCoordinatesImplementation",
 CellID->354695682],

Cell[TextData[{
 "See ",
 ButtonBox["Duffy's coordinates for multidimensional singularity handling",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#738848244"],
 " for the theory of Duffy's coordinates."
}], "Text",
 CellID->45474127],

Cell["The implementation is based on the following two theorems.", "Text",
 CellID->180672544],

Cell[TextData[{
 StyleBox["Theorem 1:",
  FontWeight->"Bold"],
 " An ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional cube can be divided into ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " disjoint geometrically equivalent ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional pyramids (with bases ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"d", "-", "1"}], ")"}], TraditionalForm]], "InlineMath"],
 "-dimensional cubes) and with coinciding apexes. "
}], "Text",
 CellID->12172749],

Cell[TextData[{
 StyleBox["Proof:",
  FontWeight->"Bold"],
 " Assume the side length of the cube is ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 ", the cube has a vertex at the origin, and the coordinates of all other \
vertexes are ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 " or 0. Consider the ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"d", "-", "1"}], ")"}], TraditionalForm]], "InlineMath"],
 "-dimensional cube walls ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["w", "s"], "=", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox["c", "1"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["c", 
       RowBox[{"s", "-", "1"}]], ",", "1", ",", " ", 
      SubscriptBox["c", 
       RowBox[{"s", "+", "1"}]], ",", "\[Ellipsis]", ",", 
      SubscriptBox["c", "d"]}], "}"}]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["c", "i"], "\[Element]", 
    RowBox[{"[", 
     RowBox[{"0", ",", "1"}], "]"}]}], TraditionalForm]], "InlineMath"],
 ". Their number is exactly ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 ", ",
 "and the origin does not belong to them. Each of the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["w", "s"], TraditionalForm]], "InlineMath"],
 " walls can form a pyramid with the origin. This proves the theorem."
}], "Text",
 CellID->80367932],

Cell["Here is a plot that illustrates the theorem in 3D.", "MathCaption",
 CellID->31870419],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"grx", "=", 
   RowBox[{"GraphicsComplex", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0", ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "0"}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Polygon", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "2", ",", "3"}], "}"}], "]"}], ",", 
       RowBox[{"Polygon", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "3", ",", "4"}], "}"}], "]"}], ",", 
       RowBox[{"Polygon", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "4", ",", "5"}], "}"}], "]"}], ",", 
       RowBox[{"Polygon", "[", 
        RowBox[{"{", 
         RowBox[{"1", ",", "5", ",", "2"}], "}"}], "]"}], ",", 
       RowBox[{"Polygon", "[", 
        RowBox[{"{", 
         RowBox[{"2", ",", "3", ",", "4", ",", "5"}], "}"}], "]"}]}], "}"}]}],
     "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"gry", "=", 
   RowBox[{"MapAt", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Map", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"RotateLeft", "[", "#", "]"}], "&"}], ",", "#"}], "]"}], 
      "&"}], ",", "grx", ",", 
     RowBox[{"{", "1", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grz", "=", 
   RowBox[{"MapAt", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Map", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"RotateRight", "[", "#", "]"}], "&"}], ",", "#"}], "]"}], 
      "&"}], ",", "grx", ",", 
     RowBox[{"{", "1", "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Graphics3D", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Opacity", "[", "0.5", "]"}], ",", "Red", ",", "grx", ",", "Cyan",
     ",", "gry", ",", "Yellow", ",", "grz"}], "}"}], "]"}]}], "Input",
 CellLabel->"In[89]:=",
 CellID->242249857],

Cell[BoxData[
 Graphics3DBox[
  {RGBColor[1, 0, 0], Opacity[0.5], 
   GraphicsComplex3DBox[{{0, 0, 0}, {1, 0, 0}, {1, 0, 1}, {1, 1, 1}, {1, 1, 
    0}}, {Polygon3DBox[{1, 2, 3}], Polygon3DBox[{1, 3, 4}], 
     Polygon3DBox[{1, 4, 5}], Polygon3DBox[{1, 5, 2}], 
     Polygon3DBox[{2, 3, 4, 5}]}], 
   {RGBColor[0, 1, 1], 
    GraphicsComplex3DBox[{{0, 0, 0}, {0, 0, 1}, {0, 1, 1}, {1, 1, 1}, {1, 0, 
     1}}, {Polygon3DBox[{1, 2, 3}], Polygon3DBox[{1, 3, 4}], 
      Polygon3DBox[{1, 4, 5}], Polygon3DBox[{1, 5, 2}], 
      Polygon3DBox[{2, 3, 4, 5}]}]}, 
   {RGBColor[1, 1, 0], 
    GraphicsComplex3DBox[{{0, 0, 0}, {0, 1, 0}, {1, 1, 0}, {1, 1, 1}, {0, 1, 
     1}}, {Polygon3DBox[{1, 2, 3}], Polygon3DBox[{1, 3, 4}], 
      Polygon3DBox[{1, 4, 5}], Polygon3DBox[{1, 5, 2}], 
      Polygon3DBox[{2, 3, 4, 5}]}]}}]], "Output",
 ImageSize->{184, 195},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[92]=",
 CellID->9317190,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[TextData[{
 "If the ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " axes are denoted ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "1"], ",", 
    SubscriptBox["x", "2"], ",", "\[Ellipsis]", ",", 
    SubscriptBox["x", "d"]}], TraditionalForm]], "InlineMath"],
 " the pyramid formed with the wall ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["w", "1"], "=", 
    RowBox[{"{", 
     RowBox[{"1", ",", 
      SubscriptBox["c", "2"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["c", "d"]}], "}"}]}], TraditionalForm]], "InlineMath"],
 " can be described as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"0", "\[LessEqual]", 
     SubscriptBox["x", "1"], "\[LessEqual]", "1"}], ",", " ", 
    RowBox[{"0", "\[LessEqual]", 
     SubscriptBox["x", "i"], "\[LessEqual]", 
     SubscriptBox["x", "1"]}], ",", " ", 
    RowBox[{"i", "\[Element]", 
     RowBox[{"{", 
      RowBox[{"2", ",", "\[Ellipsis]", ",", "d"}], "}"}], " "}]}], 
   TraditionalForm]], "InlineMath"],
 ". Let ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Sigma]", "i"], TraditionalForm]], "InlineMath"],
 " denote the permutation derived after rotating ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"1", ",", "\[Ellipsis]", ",", "d"}], "}"}], TraditionalForm]], 
  "InlineMath"],
 " cyclically ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 " times to the left (i.e. applying ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 " times ",
 Cell[BoxData[
  ButtonBox["RotateLeft",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RotateLeft"]], "InlineFormula"],
 " to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"1", ",", "\[Ellipsis]", ",", "d"}], "}"}], TraditionalForm]], 
  "InlineMath"],
 "). Then the following theorem holds:"
}], "Text",
 CellID->737331093],

Cell[TextData[{
 StyleBox["Theorem 2:",
  FontWeight->"Bold"],
 " For any integral over the unit cube the following equalities holds:"
}], "Text",
 CellID->122086428],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{"\[Ellipsis]", 
      RowBox[{
       SubsuperscriptBox["\[Integral]", "0", "1"], 
       RowBox[{
        RowBox[{"f", "(", 
         RowBox[{
          SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
          SubscriptBox["x", "d"]}], ")"}], 
        RowBox[{"\[DifferentialD]", 
         SubscriptBox["x", "1"]}], "\[Ellipsis]", 
        RowBox[{"\[DifferentialD]", 
         SubscriptBox["x", "d"]}]}]}]}]}]}], "=", 
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", 
       SubscriptBox["x", "1"]], 
      RowBox[{"\[Ellipsis]", 
       RowBox[{
        SubsuperscriptBox["\[Integral]", "0", 
         SubscriptBox["x", "1"]], 
        RowBox[{
         UnderoverscriptBox["\[Sum]", 
          RowBox[{"i", "=", "0"}], 
          RowBox[{"d", "-", "1"}]], 
         RowBox[{
          RowBox[{"f", "(", 
           RowBox[{
            SubscriptBox["x", 
             SubscriptBox[
              SuperscriptBox["\[Sigma]", "i"], "1"]], ",", "\[Ellipsis]", ",",
             
            SubscriptBox["x", 
             SubscriptBox[
              SuperscriptBox["\[Sigma]", "i"], "d"]]}], ")"}], 
          RowBox[{"\[DifferentialD]", 
           SubscriptBox["x", 
            SubscriptBox[
             SuperscriptBox["\[Sigma]", "i"], "1"]]}], "\[Ellipsis]", 
          RowBox[{"\[DifferentialD]", 
           SubscriptBox["x", 
            SubscriptBox[
             SuperscriptBox["\[Sigma]", "i"], "d"]]}]}]}]}]}]}]}], "=", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{"\[Ellipsis]", 
       RowBox[{
        SubsuperscriptBox["\[Integral]", "0", "1"], 
        RowBox[{
         SuperscriptBox[
          SubscriptBox["x", "1"], 
          RowBox[{"d", "-", "1"}]], 
         RowBox[{
          UnderoverscriptBox["\[Sum]", 
           RowBox[{"i", "=", "0"}], "d"], 
          RowBox[{
           RowBox[{"f", "(", 
            RowBox[{
             RowBox[{
              SubscriptBox["x", "1"], 
              SubscriptBox["x", 
               SubscriptBox[
                SuperscriptBox["\[Sigma]", "i"], "1"]]}], ",", "\[Ellipsis]", 
             ",", 
             SubscriptBox["x", 
              SubscriptBox[
               SuperscriptBox["\[Sigma]", "i"], 
               RowBox[{"i", "+", "1"}]]], ",", "\[Ellipsis]", ",", 
             RowBox[{
              SubscriptBox["x", "1"], 
              SubscriptBox["x", 
               SubscriptBox[
                SuperscriptBox["\[Sigma]", "i"], "d"]]}]}], ")"}], 
           RowBox[{"\[DifferentialD]", 
            SubscriptBox["x", 
             SubscriptBox[
              SuperscriptBox["\[Sigma]", "i"], "1"]]}], "\[Ellipsis]", 
           RowBox[{
            RowBox[{"\[DifferentialD]", 
             SubscriptBox["x", 
              SubscriptBox[
               SuperscriptBox["\[Sigma]", "i"], "d"]]}], 
            "."}]}]}]}]}]}]}]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->513750759],

Cell[TextData[{
 StyleBox["Proof:",
  FontWeight->"Bold"],
 " The first equality follows from Theorem 1. The second equality is just a \
change of variables that transforms a pyramid to a cube."
}], "Text",
 CellID->502612509],

Cell["\<\
Here is a function that gives the rules and the Jacobian for the \
transformation of a hypercube with a specified side into a region. \
\>", "MathCaption",
 CellID->199020356],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"FRangesToCube", "[", 
    RowBox[{"ranges_", ",", 
     RowBox[{"cubeSides", ":", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"_", ",", "_"}], "}"}], "..."}], "}"}]}]}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"Module", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"t", ",", "t1", ",", "jac", ",", "vars", ",", 
        RowBox[{"rules", "=", 
         RowBox[{"{", "}"}]}]}], "}"}], ",", "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"vars", "=", 
        RowBox[{"First", "/@", "ranges"}]}], ";", "\[IndentingNewLine]", 
       RowBox[{"t", "=", 
        RowBox[{"MapThread", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"(", 
            RowBox[{
             RowBox[{"t1", "=", 
              RowBox[{"Rescale", "[", 
               RowBox[{
                RowBox[{"#1", "[", 
                 RowBox[{"[", "1", "]"}], "]"}], ",", "#2", ",", 
                RowBox[{
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{"#1", "[", 
                    RowBox[{"[", "2", "]"}], "]"}], ",", 
                   RowBox[{"#1", "[", 
                    RowBox[{"[", "3", "]"}], "]"}]}], "}"}], "/.", 
                 "rules"}]}], "]"}]}], ";", "\[IndentingNewLine]", 
             RowBox[{"AppendTo", "[", 
              RowBox[{"rules", ",", 
               RowBox[{
                RowBox[{"#1", "[", 
                 RowBox[{"[", "1", "]"}], "]"}], "\[Rule]", "t1"}]}], "]"}], 
             ";", "t1"}], ")"}], "&"}], ",", 
          RowBox[{"{", 
           RowBox[{"ranges", ",", "cubeSides"}], "}"}]}], "]"}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"jac", "=", 
        RowBox[{"Times", "@@", 
         RowBox[{"MapThread", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"D", "[", 
             RowBox[{"#1", ",", "#2"}], "]"}], "&"}], ",", 
           RowBox[{"{", 
            RowBox[{"t", ",", "vars"}], "}"}]}], "]"}]}]}], ";", 
       "\[IndentingNewLine]", 
       RowBox[{"{", 
        RowBox[{"rules", ",", "jac"}], "}"}]}]}], "\[IndentingNewLine]", 
     "]"}], "/;", 
    RowBox[{
     RowBox[{"Length", "[", "ranges", "]"}], "\[Equal]", 
     RowBox[{"Length", "[", "cubeSides", "]"}]}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"FRangesToCube", "[", 
    RowBox[{"ranges_", ",", 
     RowBox[{"cubeSide", ":", 
      RowBox[{"{", 
       RowBox[{"_", ",", "_"}], "}"}]}]}], "]"}], ":=", 
   RowBox[{"FRangesToCube", "[", 
    RowBox[{"ranges", ",", 
     RowBox[{"Table", "[", 
      RowBox[{"cubeSide", ",", 
       RowBox[{"{", 
        RowBox[{"Length", "[", "ranges", "]"}], "}"}]}], "]"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"FRangesToCube", "[", "ranges_", "]"}], ":=", 
   RowBox[{"FRangesToCube", "[", 
    RowBox[{"ranges", ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}]}], "]"}]}], ";"}]}], "Input",
 CellLabel->"In[93]:=",
 CellID->513649867],

Cell["\<\
Here is an example of unit-square to infinite region rescaling.\
\>", "MathCaption",
 CellID->563952543],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FRangesToCube", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "8"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "x", ",", "\[Infinity]"}], "}"}]}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[96]:=",
 CellID->452674210],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"x", "\[Rule]", 
      RowBox[{"8", " ", "x"}]}], ",", 
     RowBox[{"y", "\[Rule]", 
      RowBox[{
       RowBox[{"-", "1"}], "+", 
       RowBox[{"8", " ", "x"}], "+", 
       FractionBox["1", 
        RowBox[{"1", "-", "y"}]]}]}]}], "}"}], ",", 
   FractionBox["8", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", "y"}], ")"}], "2"]]}], "}"}]], "Output",
 ImageSize->{256, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[96]=",
 CellID->221867314]
}, Open  ]],

Cell["\<\
Here is a function that computes the integrals obtained by the Duffy's \
coordinates technique when the singularity is at the origin.\
\>", "MathCaption",
 CellID->67462453],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"DuffyCoordinatesAtOrigin", "[", 
    RowBox[{"F_", ",", "ranges___"}], "]"}], ":=", 
   RowBox[{"DuffyCoordinatesBounds", "[", 
    RowBox[{"F", ",", 
     RowBox[{"First", "/@", 
      RowBox[{"{", "ranges", "}"}]}], ",", 
     RowBox[{"Transpose", "[", 
      RowBox[{"Rest", "/@", 
       RowBox[{"{", "ranges", "}"}]}], "]"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"DuffyCoordinatesBounds", "[", 
    RowBox[{"F_", ",", "vars_", ",", "bounds_"}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "rules", ",", "jac", ",", "newF", ",", "rots", ",", "res", ",", 
       "range"}], "}"}], ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"rules", ",", "jac"}], "}"}], "=", 
       RowBox[{"FRangesToCube", "[", 
        RowBox[{"Transpose", "[", 
         RowBox[{"Prepend", "[", 
          RowBox[{"bounds", ",", "vars"}], "]"}], "]"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"newF", "=", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"F", "/.", "rules"}], ")"}], "*", "jac"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"rots", "=", 
       RowBox[{"NestList", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"RotateLeft", "[", "#1", "]"}], "&"}], ",", "vars", ",", 
         RowBox[{
          RowBox[{"Length", "[", "vars", "]"}], "-", "1"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"res", "=", 
       RowBox[{"Prepend", "[", 
        RowBox[{
         RowBox[{"Map", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"newF", "/.", 
             RowBox[{"Thread", "[", 
              RowBox[{"vars", "->", "#1"}], "]"}]}], "&"}], ",", 
           RowBox[{"Rest", "[", "rots", "]"}]}], "]"}], ",", "newF"}], 
        "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"range", "=", 
       RowBox[{"Join", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"vars", "[", 
             RowBox[{"[", "1", "]"}], "]"}], ",", "0", ",", "1"}], "}"}], 
          "}"}], ",", 
         RowBox[{"Map", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"{", 
             RowBox[{"#", ",", "0", ",", 
              RowBox[{"vars", "[", 
               RowBox[{"[", "1", "]"}], "]"}]}], "}"}], "&"}], ",", 
           RowBox[{"Rest", "[", "vars", "]"}]}], "]"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"rules", ",", "jac"}], "}"}], "=", 
       RowBox[{"FRangesToCube", "[", "range", "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"Total", "[", "res", "]"}], "/.", "rules"}], ")"}], "*", 
         "jac"}], ",", 
        RowBox[{"Sequence", " ", "@@", 
         RowBox[{"(", " ", 
          RowBox[{
           RowBox[{
            RowBox[{"{", 
             RowBox[{"#1", ",", " ", "0", ",", " ", "1"}], "}"}], " ", "&"}], 
           "  ", "/@", " ", "vars"}], ")"}]}]}], "}"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[97]:=",
 CellID->100917235],

Cell["\<\
Here is a function that computes the integrals obtained by the Duffy's \
coordinates technique for a specified corner of the hypercube where the \
singularity occurs.\
\>", "MathCaption",
 CellID->697138430],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"DuffyCoordinates", "[", 
    RowBox[{"F_", ",", "ranges___"}], "]"}], ":=", 
   RowBox[{"DuffyCoordinates", "[", 
    RowBox[{"F", ",", "ranges", ",", 
     RowBox[{"Table", "[", 
      RowBox[{"0", ",", 
       RowBox[{"{", 
        RowBox[{"Length", "[", 
         RowBox[{"{", "ranges", "}"}], "]"}], "}"}]}], "]"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"DuffyCoordinates", "[", 
    RowBox[{"F_", ",", "rangesSeq__", ",", 
     RowBox[{"corner_", "?", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"VectorQ", "[", 
         RowBox[{"#1", ",", "IntegerQ"}], "]"}], "&"}], ")"}]}]}], "]"}], ":=",
    "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"factor", ",", 
       RowBox[{"ranges", "=", 
        RowBox[{"{", "rangesSeq", "}"}]}], ",", "newrange", ",", "t"}], "}"}],
      ",", "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"factor", "=", "1"}], ";", "\[IndentingNewLine]", 
      RowBox[{"newrange", "=", 
       RowBox[{"{", "}"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"MapIndexed", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"(", "\[IndentingNewLine]", 
          RowBox[{
           RowBox[{"t", "=", 
            RowBox[{"ranges", "\[LeftDoubleBracket]", 
             RowBox[{
             "#2", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
             "\[RightDoubleBracket]"}]}], ";", "\[IndentingNewLine]", 
           RowBox[{"If", "[", 
            RowBox[{
             RowBox[{"#1", "\[Equal]", "0"}], ",", "\[IndentingNewLine]", 
             RowBox[{"newrange", "=", 
              RowBox[{"Append", "[", 
               RowBox[{"newrange", ",", "t"}], "]"}]}], ",", 
             "\[IndentingNewLine]", 
             RowBox[{
              RowBox[{"newrange", "=", 
               RowBox[{"Append", "[", 
                RowBox[{"newrange", ",", 
                 RowBox[{"{", 
                  RowBox[{
                   RowBox[{
                   "t", "\[LeftDoubleBracket]", "1", 
                    "\[RightDoubleBracket]"}], ",", 
                   RowBox[{
                   "t", "\[LeftDoubleBracket]", "3", 
                    "\[RightDoubleBracket]"}], ",", 
                   RowBox[{
                   "t", "\[LeftDoubleBracket]", "2", 
                    "\[RightDoubleBracket]"}]}], "}"}]}], "]"}]}], ";", 
              RowBox[{"factor", "=", 
               RowBox[{"-", "factor"}]}]}]}], "]"}]}], ")"}], "&"}], ",", 
        "\[IndentingNewLine]", "corner"}], "]"}], ";", "\[IndentingNewLine]", 
      
      RowBox[{"DuffyCoordinatesAtOrigin", "[", 
       RowBox[{
        RowBox[{"factor", "*", "F"}], ",", 
        RowBox[{"Sequence", "@@", "newrange"}]}], "]"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]}], "Input",
 CellLabel->"In[99]:=",
 CellID->698169174],

Cell["Here is a symbolic example.", "MathCaption",
 CellID->184347267],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DuffyCoordinates", "[", 
  RowBox[{
   RowBox[{"F", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[101]:=",
 CellID->416495380],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"x", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"F", "[", 
       RowBox[{"x", ",", 
        RowBox[{"x", " ", "y"}]}], "]"}], "+", 
      RowBox[{"F", "[", 
       RowBox[{
        RowBox[{"x", " ", "y"}], ",", "x"}], "]"}]}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{305, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[101]=",
 CellID->572655054]
}, Open  ]],

Cell["Here is another symbolic example.", "MathCaption",
 CellID->2285240],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"DuffyCoordinates", "[", 
  RowBox[{
   RowBox[{"F", "[", 
    RowBox[{"x", ",", "y", ",", "z"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[102]:=",
 CellID->77448927],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    SuperscriptBox["x", "2"], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"F", "[", 
       RowBox[{"x", ",", 
        RowBox[{"x", " ", "y"}], ",", 
        RowBox[{"x", " ", "z"}]}], "]"}], "+", 
      RowBox[{"F", "[", 
       RowBox[{
        RowBox[{"x", " ", "y"}], ",", 
        RowBox[{"x", " ", "z"}], ",", "x"}], "]"}], "+", 
      RowBox[{"F", "[", 
       RowBox[{
        RowBox[{"x", " ", "z"}], ",", "x", ",", 
        RowBox[{"x", " ", "y"}]}], "]"}]}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "0", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{532, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[102]=",
 CellID->809666225]
}, Open  ]],

Cell["Here is a computational example.", "MathCaption",
 CellID->382244866],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "@@", 
  RowBox[{"DuffyCoordinates", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"], "+", 
       SuperscriptBox["z", "2"]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"z", ",", "0", ",", "2"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[103]:=",
 CellID->28352571],

Cell[BoxData["9.528129056539287`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[103]=",
 CellID->10146926]
}, Open  ]],

Cell["\<\
Using Duffy's coordinates is much faster than using no singularity handling \
(see the next example).\
\>", "MathCaption",
 CellID->116601970],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"res", "=", 
   RowBox[{"NINT", "@@", 
    RowBox[{"DuffyCoordinates", "[", 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"3", "-", "y"}], ")"}], "2"], "+", 
         SuperscriptBox["z", "2"]}]]], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "4"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "3"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"z", ",", "0", ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"res", "=", 
   RowBox[{
    RowBox[{"Hold", "[", 
     RowBox[{"Evaluate", "[", "res", "]"}], "]"}], "/.", 
    RowBox[{"NINT", "\[Rule]", "NIntegrate"}]}]}], ";"}], "\n", 
 RowBox[{"Timing", " ", "@@", "res"}]}], "Input",
 CellLabel->"In[108]:=",
 CellID->7452570],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.009998000000005745`", ",", "9.528129056539287`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[110]=",
 CellID->309252236]
}, Open  ]],

Cell["\<\
Integration using no singularity handling.\[LineSeparator]\
\>", "MathCaption",
 CellID->93705055],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", " ", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"], "+", 
       SuperscriptBox["z", "2"]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"z", ",", "0", ",", "2"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "None"}]}], "}"}]}]}],
    "]"}]}]], "Input",
 CellLabel->"In[111]:=",
 CellID->38321334],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.18097100000000596`", ",", "9.528129425758879`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[111]=",
 CellID->270297002]
}, Open  ]],

Cell[TextData[{
 "Of course, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s internal implementation gives similar performance results."
}], "MathCaption",
 CellID->590800148],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Timing", " ", "@", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"3", "-", "y"}], ")"}], "2"], "+", 
       SuperscriptBox["z", "2"]}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "4"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", "0", ",", "3"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"z", ",", "0", ",", "2"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<DuffyCoordinates\>\"", ",", 
       RowBox[{"\"\<Corners\>\"", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], ",", 
       RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[107]:=",
 CellID->324071473],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.011997999999993222`", ",", "9.528129056539287`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[107]=",
 CellID->646574165]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Ignoring the singularity", "Subsection",
 CellTags->"IgnoringSingularity",
 CellID->74051012],

Cell[TextData[{
 "Another way of handling a singularity is to ignore it. ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " ignores sampling points that coincide with a singular point. "
}], "Text",
 CellID->605385],

Cell[TextData[{
 "Consider the following integral that has a singular point at ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"]
}], "Text",
 CellID->665342016],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubsuperscriptBox["\[Integral]", "0", "2"], 
   RowBox[{
    RowBox[{"log", "(", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", "x"}], ")"}], "2"], ")"}], 
    RowBox[{
     RowBox[{"\[DifferentialD]", "x"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->60311523],

Cell[TextData[{
 "The integrand goes to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 " ",
 "when the integration variable is close to ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->83591884],

Cell["Here is a plot of the integrand.", "MathCaption",
 CellID->27619765],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]], "Input",
 CellLabel->"In[118]:=",
 CellID->213756822],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{241, 141},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[118]=",
 CellID->81668348]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " gives a result that is close to the exact one."
}], "MathCaption",
 CellID->529829003],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"exact", "=", 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "2"], 
      RowBox[{
       RowBox[{"Log", "[", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}]}], ";"}], "\[IndentingNewLine]", 
   
   FractionBox[
    RowBox[{"exact", "-", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"Log", "[", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]}], 
    "exact"]}]}]], "Input",
 CellLabel->"In[114]:=",
 CellID->566767169],

Cell[BoxData["0.000012401659211791483`"], "Output",
 ImageSize->{88, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[115]=",
 CellID->90906967]
}, Open  ]],

Cell[TextData[{
 "Convergence is achieved when ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " is increased."
}], "MathCaption",
 CellID->419340562],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Log", "[", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<GlobalAdaptive\>\""}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "100"}]}], "]"}]], "Input",
 CellLabel->"In[45]:=",
 CellID->27729582],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[45]:=",
 CellID->735891818],

Cell[BoxData[
 RowBox[{"-", "3.999999418672229`"}]], "Output",
 ImageSize->{26, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[45]=",
 CellID->1003837792]
}, Open  ]],

Cell[TextData[{
 "With its default options ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " has a sampling point at ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 ", as it can be seen from the following."
}], "Text",
 CellID->387451739],

Cell[TextData[{
 "Check that ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " has 1 as a sampling point."
}], "MathCaption",
 CellID->4722665],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"InputForm", "/@", 
    RowBox[{"Select", "[", 
     RowBox[{"#", ",", 
      RowBox[{
       RowBox[{"0.9", "<", "#", "<", "1.01"}], "&"}]}], "]"}]}], "&"}], "@", 
  RowBox[{
   RowBox[{"Reap", "[", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{"x", ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"2", ",", "1"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[119]:=",
 CellID->10608136],

Cell[BoxData[
 RowBox[{"{", 
  InterpretationBox[
   StyleBox["1.",
    ShowStringCharacters->True,
    NumberMarks->True],
   InputForm[1.],
   AutoDelete->True,
   Editable->True], "}"}]], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[119]=",
 CellID->273115485]
}, Open  ]],

Cell[TextData[{
 "But for ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["NIntegrate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NIntegrate"], "[", 
   RowBox[{
    RowBox[{
     ButtonBox["Log",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Log"], "[", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"1", "-", 
        StyleBox["x", "TI"]}], ")"}], "2"], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", "0", ",", "2"}], "}"}]}], "]"}]], 
  "InlineFormula"],
 " the evaluation monitor has not picked a sampling point that is ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->6024480],

Cell[TextData[{
 "Sampling points that belong to the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"1", "-", 
      SuperscriptBox["10", "6"]}], ",", 
     RowBox[{"1", "+", 
      SuperscriptBox["10", "6"]}]}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->31653682],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"InputForm", "/@", 
    RowBox[{"Select", "[", 
     RowBox[{"#", ",", 
      RowBox[{
       RowBox[{"0.99999", "<", "#", "<", "1.00001"}], "&"}]}], "]"}]}], "&"}],
   "@", 
  RowBox[{
   RowBox[{"Reap", "[", "\[IndentingNewLine]", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"Log", "[", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"2", ",", "1"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[120]:=",
 CellID->449599167],

Cell[BoxData[
 RowBox[{"{", "}"}]], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[120]=",
 CellID->4182420]
}, Open  ]],

Cell[TextData[{
 "In other words, the singularity at ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 " is ignored. Ignoring the singularity is equivalent to having an integrand \
that is zero at the singular sampling point."
}], "Text",
 CellID->522468040],

Cell[TextData[{
 "Note that the integral is easily integrated if the singular point is \
specified in the variable range. Following are the numbers of sampling points \
and timings for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with the singular and nonsingular range specifications."
}], "Text",
 CellID->73396471],

Cell["Integration with the singular point specified.", "MathCaption",
 CellID->83061034],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"Log", "[", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "1", ",", "2"}], "}"}], ",", 
        RowBox[{"EvaluationMonitor", ":>", 
         RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
     RowBox[{"[", 
      RowBox[{"2", ",", "1"}], "]"}], "]"}], "//", "Length"}], ",", 
   RowBox[{
    RowBox[{"Timing", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"Log", "[", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1", ",", "2"}], "}"}]}], "]"}], "]"}], 
    "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[123]:=",
 CellID->42596938],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"260", ",", "0.005999000000003536`"}], "}"}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[123]=",
 CellID->616006260]
}, Open  ]],

Cell["Integration by ignoring the singularity.", "MathCaption",
 CellID->30052593],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"Log", "[", 
         SuperscriptBox[
          RowBox[{"(", 
           RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
        RowBox[{"MaxRecursion", "->", "20"}], ",", 
        RowBox[{"EvaluationMonitor", ":>", 
         RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
     RowBox[{"[", 
      RowBox[{"2", ",", "1"}], "]"}], "]"}], "//", "Length"}], ",", 
   RowBox[{
    RowBox[{"Timing", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"Log", "[", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"1", "-", "x"}], ")"}], "2"], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "2"}], "}"}], ",", 
       RowBox[{"MaxRecursion", "->", "20"}]}], "]"}], "]"}], "[", 
    RowBox[{"[", "1", "]"}], "]"}]}], "}"}]], "Input",
 CellLabel->"In[122]:=",
 CellID->169518830],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"670", ",", "0.008998000000003205`"}], "}"}]], "Output",
 ImageSize->{107, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[122]=",
 CellID->947789399]
}, Open  ]],

Cell["\<\
A more interesting example of ignoring the singularity is using Bessel \
functions in the denominator of the integrand.\
\>", "Text",
 CellID->397337811],

Cell["Integral with several (five) integrable singularities.", "MathCaption",
 CellID->694091574],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{"Sqrt", "[", 
      RowBox[{"Abs", "[", 
       RowBox[{"BesselJ", "[", 
        RowBox[{"2", ",", "x"}], "]"}], "]"}], "]"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "20"}], "}"}], ",", 
    RowBox[{"MaxRecursion", "\[Rule]", "1000"}]}], "]"}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[124]:=",
 CellID->460285200],

Cell["59.539197071142375", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[124]//InputForm=",
 CellID->498727177]
}, Open  ]],

Cell[TextData[{
 "The result can be checked using ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with singular range specification with the zeros of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"], "[", 
   RowBox[{"2", ",", 
    StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
 " (see ",
 Cell[BoxData[
  ButtonBox["BesselJZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJZero"]], "InlineFormula"],
 ")."
}], "Text",
 CellID->280304199],

Cell["\<\
Integration with the Bessel zeros specified as singular points.\
\>", "MathCaption",
 CellID->78255370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{"Sqrt", "[", 
      RowBox[{"Abs", "[", 
       RowBox[{"BesselJ", "[", 
        RowBox[{"2", ",", "x"}], "]"}], "]"}], "]"}]], ",", 
    RowBox[{"{", 
     RowBox[{
     "x", ",", "1", ",", "5.135622301840683`", ",", "8.417244140399848`", ",",
       "11.619841172149059`", ",", "14.79595178235126`", ",", 
      "17.959819494987826`", ",", "20"}], "}"}], ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "8"}]}], "]"}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[125]:=",
 CellID->164525516],

Cell["59.53926944377681", "Output",
 ImageSize->{119, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[125]//InputForm=",
 CellID->221037452]
}, Open  ]],

Cell[TextData[{
 "Needless to say, the last integration required the calculation of the ",
 Cell[BoxData[
  ButtonBox["BesselJ",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJ"]], "InlineFormula"],
 " zeros. The former one \"just integrates\" without any integrand analysis. \
"
}], "Text",
 CellID->33548582],

Cell["\<\
Ignoring the singularity may not work with oscillating integrands. \
\>", "Text",
 CellID->44233269],

Cell["For example, these two integrals are equivalent.", "MathCaption",
 CellID->27945464],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", "x"], 
     RowBox[{"Sin", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}], "==", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     FractionBox["1", "x"], 
     RowBox[{"Sin", "[", 
      FractionBox["1", "x"], "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[126]:=",
 CellID->978020674],

Cell[BoxData["True"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[126]=",
 CellID->531886725]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can do the first one."
}], "MathCaption",
 CellID->544601575],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", "x"], 
    RowBox[{"Sin", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[127]:=",
 CellID->20280246],

Cell[BoxData["0.6247132601585286`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[127]=",
 CellID->166964506]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " cannot do the second one."
}], "MathCaption",
 CellID->128446164],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox["1", "x"], 
    RowBox[{"Sin", "[", 
     FractionBox["1", "x"], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<GlobalAdaptive\>\""}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "100"}]}], "]"}]], "Input",
 CellLabel->"In[128]:=",
 CellID->95532555],

Cell[BoxData["0``-1.7166466084827094"], "Output",
 ImageSize->{49, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[128]=",
 CellID->103054574]
}, Open  ]],

Cell["\<\
However, if the integrand is monotonic in a neighborhood of its singularity, \
or more precisely, if it can be majorized by a monotonic integrable function, \
it can be shown that by ignoring the singularity, convergence will be \
reached.\
\>", "Text",
 CellID->77440787],

Cell[TextData[{
 "For theoretical justification and practical recommendations of ignoring the \
singularity see [",
 ButtonBox["DavRab65IS",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#34491249"],
 "] and [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "]."
}], "Text",
 CellID->15350963]
}, Open  ]],

Cell[CellGroupData[{

Cell["Automatic singularity handling ", "Subsection",
 CellTags->"AutomaticSingHandling",
 CellID->4135167],

Cell[CellGroupData[{

Cell["One-dimensional integration ", "Subsubsection",
 CellID->282959198],

Cell[TextData[{
 "When the option ",
 Cell[BoxData["\"\<SingularityHandler\>\""], "InlineFormula"],
 " is set to ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " for a one-dimensional integral, ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " is used on regions that are obtained by ",
 Cell[BoxData["\"\<SingularityDepth\>\""], "InlineFormula"],
 " number of partitionings. As explained ",
 ButtonBox["earlier,",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#23700706"],
 " this region will not be partitioned further as long as the ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " singularity handler works over it. If the error estimate computed by ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " does not evolve in a way predicted by the theory of the ",
 ButtonBox["double-exponential quadrature",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#526196975"],
 ", the singularity handling for this region is switched to ",
 Cell[BoxData["IMT"], "InlineFormula"],
 ". "
}], "Text",
 CellID->53300836],

Cell[TextData[{
 "As explained in ",
 ButtonBox["Convergence rate",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#175923852"],
 ", the following dependency of the error is expected with respect to the \
number of double-exponential sampling points:"
}], "Text",
 CellID->60790206],

Cell[BoxData[
 FormBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"c", " ", "n"}], 
      RowBox[{"log", " ", "n"}]]}]], ","}], TraditionalForm]], "DisplayMath",
 CellID->213862900],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 " is a positive constant. Consider the relative errors ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["E", "m"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["E", "n"], TraditionalForm]], "InlineMath"],
 " of two consecutive ",
 "double-exponential",
 " quadrature calculations, made with ",
 Cell[BoxData[
  FormBox["m", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " number of sampling points respectively, for which ",
 Cell[BoxData[
  FormBox[
   RowBox[{"m", "<", "n"}], TraditionalForm]], "InlineMath"],
 ". Assuming ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["E", "m"], "<", "1"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["E", "n"], "<", "1"}], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["E", "m"], ">", 
    SubscriptBox["E", "n"]}], TraditionalForm]], "InlineMath"],
 " it should be expected that"
}], "Text",
 CellID->59327555],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FractionBox[
     FormBox[
      SubscriptBox["E", "m"],
      TraditionalForm], 
     FormBox[
      SubscriptBox["E", "n"],
      TraditionalForm]], "\[GreaterEqual]", " ", 
    FractionBox[
     FormBox[
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"-", 
        FractionBox[
         RowBox[{"c", " ", "m"}], 
         RowBox[{"log", " ", "m"}]]}]],
      TraditionalForm], 
     FormBox[
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"-", 
        FractionBox[
         RowBox[{"c", " ", "n"}], 
         RowBox[{"log", " ", "n"}]]}]],
      TraditionalForm]]}], "\[DoubleLeftRightArrow]"}], 
  TraditionalForm]], "NumberedEquation",
 CellID->226870002],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FractionBox[
     FormBox[
      RowBox[{"log", " ", 
       SubscriptBox["E", "m"]}],
      TraditionalForm], 
     FormBox[
      RowBox[{"log", " ", 
       SubscriptBox["E", "n"]}],
      TraditionalForm]], "\[LessEqual]", 
    FractionBox[
     FormBox[
      RowBox[{"-", 
       FractionBox[
        RowBox[{"c", " ", "m"}], 
        RowBox[{"log", " ", "m"}]]}],
      TraditionalForm], 
     FormBox[
      RowBox[{"-", 
       FractionBox[
        RowBox[{"c", " ", "n"}], 
        RowBox[{"log", " ", "n"}]]}],
      TraditionalForm]]}], "=", 
   RowBox[{
    FractionBox[
     RowBox[{"m", " ", "log", " ", "n"}], 
     RowBox[{"n", " ", "log", " ", "m"}]], "."}]}], 
  TraditionalForm]], "NumberedEquation",
 CellID->523900344],

Cell[TextData[{
 "The switch from ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " to ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " happens when:\n(i) the region error estimate is larger than the absolute \
value of the region integral estimate (hence the relative error is not \
smaller than ",
 Cell[BoxData[
  FormBox["1", TraditionalForm]], "InlineMath"],
 ");\n(ii) the inequality (2) is not true in two different instances;\n(iii) \
the integrand values calculated with the double-exponential transformation do \
not decay fast enough."
}], "Text",
 CellID->57077604],

Cell[TextData[{
 "Here is an example of a switch from ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " to ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " singularity handling. On the plot the integrand is sampled at the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " coordinates in the order of the ",
 Cell[BoxData[
  StyleBox["y", "TI"]], "InlineFormula"],
 " coordinates. The patterns of the sampling points over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", 
     FractionBox["1", "16"]}], "]"}], TraditionalForm]], "InlineMath"],
 " show the change from Gaussian quadrature (",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[Element]", 
    RowBox[{"[", 
     RowBox[{"0", ",", "97"}], "]"}]}], TraditionalForm]], "InlineMath"],
 ") to ",
 "double-exponential",
 " quadrature (",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[Element]", 
    RowBox[{"[", 
     RowBox[{"98", ",", "160"}], "]"}]}], TraditionalForm]], "InlineMath"],
 "), which later is replaced by Gaussian quadrature using the ",
 ButtonBox["IMT variable transformation ",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#150310618"],
 "(",
 Cell[BoxData[
  FormBox[
   RowBox[{"y", "\[Element]", 
    RowBox[{"[", 
     RowBox[{"160", ",", "400"}], "]"}]}], TraditionalForm]], "InlineMath"],
 "). "
}], "MathCaption",
 CellID->297705820],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"k", "=", "0"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   FractionBox["70", 
    RowBox[{
     RowBox[{
      SuperscriptBox["10", "4"], " ", 
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{"x", "-", 
         FractionBox["1", "32"]}], ")"}], "2"]}], "+", 
     FractionBox["1", "16"]}]]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"f", "[", "x", "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"Point", "[", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"N", "[", "x", "]"}], ",", 
            RowBox[{"k", "++"}]}], "}"}], "]"}], "]"}]}]}], "]"}], "]"}], "[",
     
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";", 
  RowBox[{"Graphics", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", "0.006", "]"}], ",", "gr"}], "}"}], ",", 
    RowBox[{"AspectRatio", "\[Rule]", "1"}], ",", 
    RowBox[{"Axes", "\[Rule]", "True"}], ",", 
    RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
    RowBox[{"GridLines", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"None", ",", 
       RowBox[{"{", 
        RowBox[{"97", ",", "160"}], "}"}]}], "}"}]}]}], "]"}]}]}], "Input",
 CellLabel->"In[143]:=",
 CellID->322013271],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFWUtv1DAQdpPsQ4WqtNsCAiR6oBcQVyQkzoBASBwK91VVqT3wKhVC4rIV
Ai68Cirl0XaXE2ek/r/FTuzEHn9OnMfSSptkPOPPn2c8Yye9199aX3vU39pY
7S/d2uw/Xd9Yfb5088kmbwqnGGNH/Pf3LOPPY/4U/47GY37Xr1f4NeCqUEih
sBGtLH5c0HWtRJf8CYtF0FM8Bh460RQAXWzPLotrlGkYaxF2PcuiTdDmLYvI
arFRbJtFghtaFoGDsx+jyJPHZLl2SM85PJoWgYVKnEUcRZxPT5CpGgMzDD25
V/dlGT/YTFpk3CbjrnK9Le8qS6lPe1q7Yon6KTnJWr2qzBPEYenWWX79rWZr
2VAvRQV6KgeWh2i20opTd8Tj7g/tLb9eKJBPEnmGyLTezyG/a/J5L1ZviNxF
rOJeAbP3lS9EbsMxaA5Q/1H9a4hyqaDXY4PpdcQsx5sP4Jh3CmTs07cF86Vj
dyHKs4LILMFeKyV9fQ2i3B3THL5v2TGwHn5CNOp763RC5PcpiojkmQLrbTjm
RdqLP76ClrfJnFag1Y2Sc3gIUa6WRFmDKMtWdF4YHlsm9i8zlFY6TznbA6Nn
vFraOnJy3YU89IolsD5Dq1mJ3XOzonGzWOLInbP8gNdCD2K+g7anJNsPhl9G
qSRM5aYcA+1DlEBauvNBoO45GeBsoLuDED8lVCI5KaayUp0d0qhqfadJ3OIq
DvbtgNjp8mAwYIG4CHkQ/7EOvwhxh9+jCpjKc5YMVuReDXwhmjGoz5PW1d0G
cIUe46oVqBZjXrSHcP3gVvom8h1aYU4qV+g+ZWr13DmA6JgZ3TN/1OCfn3/u
2TXhg6IKPIKjlYlh+VZRoUOes2L8/TR3ha1YuoewZ2Aw9/euek/EkVGV65fl
L2ngHDU/HlT7DWLQzHHPQJe1mVhcaXbn2Sbzijzqt+kleeg2FlXZ6qVGSvar
6lVVHoizjhKRfgHz4ufwUJNYaGeptENZZ+AyFdrvFJ2z7pyR9K+iOj/P7JuE
LtJ1cT2C8/DzmNsKfT/VMDxZDCfUGkCfJGspq8aiQ9bf3DsK5tegJz2tcjyq
vitMLtp556s6uYatQo+Mx3lYqbJNBDUE7XZdc53NfSpmVmfqn73/qzdd0e2Q
qiZH0WqpMcdDq53mt851mLYWZf3IYYnR6tetPAbNVkW/92L1teEj1JpvzYrx
jLzvwD707EjXCf7q4fcW0ZUjY+201H6F2hNE1tjn7LDoK+7x7Pck78cgp/x6
+uZ2zok7CZdVyMqdJen4npUxy9MmamvTSPlR+hPb2P/5ZlP/APrVJIQ=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 177},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[145]=",
 CellID->587698169]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Multidimensional integration", "Subsubsection",
 CellID->416363467],

Cell[TextData[{
 "When the option ",
 Cell[BoxData["\"\<SingularityHandler\>\""], "InlineFormula"],
 " is set to ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " for a ",
 "multidimensional",
 " integral, both ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " are used. "
}], "Text",
 CellID->329145048],

Cell[TextData[{
 "A region needs to meet the following conditions in order for ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " to be applied:"
}], "Text",
 CellID->634862999],

Cell[TextData[{
 "the region is obtained by ",
 Cell[BoxData["\"\<SingularityDepth\>\""], "InlineFormula"],
 " number of bisections (or partitionings) along each axis;"
}], "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{54.3125, Inherited}, {Inherited, Inherited}},
 TextAlignment->AlignmentMarker,
 CellID->612447040],

Cell["\<\
the region is a corner of one of the initial integration regions (the \
specified integration region can be partitioned into integration regions by \
piecewise handling or by user-specified singularities).\
\>", "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{54.3125, Inherited}, {Inherited, Inherited}},
 TextAlignment->AlignmentMarker,
 CellID->65822541],

Cell[TextData[{
 "A region needs to meet the following conditions in order for ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " to be applied:"
}], "Text",
 CellID->6925048],

Cell[TextData[{
 "the region is obtained with by ",
 Cell[BoxData["\"\<SingularityDepth\>\""], "InlineFormula"],
 " number of bisections (or partitionings) along predominantly one axis;"
}], "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{54.3125, Inherited}, {Inherited, Inherited}},
 TextAlignment->AlignmentMarker,
 CellID->8725882],

Cell["\<\
the region is not a corner region and it is on a side of one of the initial \
integration regions.\
\>", "Text",
 CellDingbat->"\[FilledSmallSquare]",
 CellMargins->{{54.3125, Inherited}, {Inherited, Inherited}},
 TextAlignment->AlignmentMarker,
 CellID->141480320],

Cell[TextData[{
 "In other words, ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " is applied to regions that are derived through ",
 Cell[BoxData["\"\<SingularityDepth\>\""], "InlineFormula"],
 " number of partitionings but do not satisfy the conditions of the ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " automatic application."
}], "Text",
 CellID->144173837],

Cell[TextData[{
 Cell[BoxData["IMT"], "InlineFormula"],
 " is effective if the singularity is along one of the axes. Using ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " for point singularities can be counterproductive."
}], "Text",
 CellID->295322025],

Cell[TextData[{
 "Sampling points of ",
 "two-dimensional",
 " integration, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{"x", "+", "y"}]]], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", with ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " (left) and ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 " (right) singularity handling. It can be seen that the automatic \
singularity handling uses almost two times more points than ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 ". To illustrate the effect of the singularity handlers they are applied \
after two bisections."
}], "MathCaption",
 CellID->647021145],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pointsAutomatic", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        SqrtBox[
         RowBox[{"x", "+", "y"}]]], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
          RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", "2"}], ",", 
          RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "Automatic"}]}], 
         "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], 
    "\[LeftDoubleBracket]", 
    RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}], ";", 
  RowBox[{"pointsDuffy", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        SqrtBox[
         RowBox[{"x", "+", "y"}]]], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
          RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", "2"}], ",", 
          RowBox[{
          "\"\<SingularityHandler\>\"", "\[Rule]", 
           "\"\<DuffyCoordinates\>\""}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], 
    "\[LeftDoubleBracket]", 
    RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}], ";"}], "\n", 
 RowBox[{"Row", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"Graphics", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"PointSize", "[", "0.015", "]"}], ",", " ", 
        RowBox[{"Point", " ", "/@", " ", "pointsAutomatic"}]}], "}"}], ",", 
      " ", 
      RowBox[{"Axes", " ", "->", " ", "True"}], ",", 
      RowBox[{"ImageSize", "\[Rule]", "200"}], ",", 
      RowBox[{"PlotLabel", "\[Rule]", 
       RowBox[{"\"\<Sampling\\ Points:\\ \>\"", "<>", 
        RowBox[{"ToString", "[", 
         RowBox[{"Length", "[", "pointsAutomatic", "]"}], "]"}]}]}]}], "]"}], 
    ",", " ", "\n", "   ", 
    RowBox[{"Graphics", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"PointSize", "[", "0.015", "]"}], ",", " ", 
        RowBox[{"Point", " ", "/@", " ", "pointsDuffy"}]}], "}"}], ",", " ", 
      RowBox[{"Axes", " ", "->", " ", "True"}], ",", 
      RowBox[{"ImageSize", "\[Rule]", "200"}], ",", 
      RowBox[{"PlotLabel", "\[Rule]", 
       RowBox[{"\"\<Sampling\\ Points:\\ \>\"", "<>", 
        RowBox[{"ToString", "[", 
         RowBox[{"Length", "[", "pointsDuffy", "]"}], "]"}]}]}]}], "]"}]}], 
   "}"}], "]"}]}], "Input",
 CellLabel->"In[133]:=",
 CellID->357005237],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{400, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[134]=",
 CellID->96085761]
}, Open  ]],

Cell[TextData[{
 "Timings for the integral, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      FractionBox["1", 
       SqrtBox[
        RowBox[{"x", "+", "y"}]]], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", with singularity handlers ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 ", and ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " and with no singularity handling. The integral has a point singularity at \
",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{"0", ",", "0"}], "}"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->63282197],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"#", ",", 
       RowBox[{
        RowBox[{"Timing", "[", 
         RowBox[{"NIntegrate", "[", 
          RowBox[{
           FractionBox["1", 
            SqrtBox[
             RowBox[{"x", "+", "y"}]]], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"Method", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
              RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "#"}]}], 
             "}"}]}]}], "]"}], "]"}], "[", 
        RowBox[{"[", "1", "]"}], "]"}]}], "}"}], "&"}], "/@", 
    RowBox[{"{", 
     RowBox[{
     "Automatic", ",", "\"\<DuffyCoordinates\>\"", ",", "\"\<IMT\>\"", ",", 
      "None"}], "}"}]}], ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"None", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"StyleForm", "[", 
         RowBox[{
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{"\"\<Singularity\>\"", ",", " ", "\"\<handler\>\""}], 
            "}"}], "]"}], ",", 
          RowBox[{"FontFamily", "\[Rule]", "Times"}]}], "]"}], ",", 
        RowBox[{"StyleForm", "[", 
         RowBox[{"\"\<Time (s)\>\"", ",", 
          RowBox[{"FontFamily", "\[Rule]", "Times"}]}], "]"}]}], "}"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[47]:=",
 CellID->460510895],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[47]:=",
 CellID->861976832],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWttu1DAQdXPZLW2BqghouYh+AeId9QmJchF8AVBWZUX7gIC2P8B7oY/8
Rv/QxImdjI/HXnuT0CI1WU/s8VzPTBzezc6O5t9mZ8eHs939k9mPo+PD092X
308qUr4ihLis/j5ui+q3rH7Vf5dSVnf3mlcPRV7/FOanomX6LoruWUMtqvuB
fqoeZtW9JONccykJF0MvCfeOt6z/ZYZetiS9dYCat1TxmZ2Va85FKmeHWsJ+
PEe1n3LBSjOetBq21+F1vMFSJ/65qAWMh5VfET+lyk0kQhsoCxfa0oWev6zE
bcwVHX+dDmTHDwnSm4zgrZ/zexV0HskPkpeZd14m7Jkm1w562LxYJKehsPoz
khPePxPkSs6VSCp6uLG3i3en2pbr0sax0L1oY9GHue7VQmB1KakURliDpl/l
EOjL2oD3J5k199rz6lB2SE3Gw9KQlO21v5RLI+f48g2JjZy0HBIOZcu+iOiV
t8t0SfLC4vslQe7/i5iLcWutus/leCj6WM+9qEdPQMo/NbXQgmWdO7R4BiVL
Jlpyj3QGXc09HrvNKsOlEF1HvSF5TM+ZdVQajFPdBWkZAv5jaGgBYzM+rvtE
TyBTFuJ4YIcpUGUIvde8fPyVLE5Ctoah9q7P8k5oT08ah1J5pznr4evls4Bd
VrlxQq2ha0OW4sa0wxjYP9Y6rNFX5SHaGyBPij/rfX1g1XPajZSjWDyAXT2s
njHasTQZfMfkkSNSDh6/4jxBq0lKZx7nHyEMF6M7rbN8faS8JtKtuNjbtTTr
DMk3m6vdtNPoa8vIiGOjlZxdOFV7HXjj3rzcD2HWPRjvXJG2PLXRXXnhprT9
rrRT9AeSz7in0u0rTe9kkIXr3FLfzt1zAe5tAzs0NWcq3M4Px3uM9D06vyBm
GrlD3VL9bJWuotdAJYp7Z3BmPfeuHfFEIbobq2UJ652UDZ2d4zii9Fi59lgu
fC4j7+kCyyzuiag08XGQYiVfXzKkXfr2KeNboUesROYbbxnkaLKyX9+A9qKd
Au0YnG9RBNsoKtKzToP9KVoufdYDa/xnaWsw3oHxmwRpxzrhCVVn45lnIPcL
mXayE6jAarNlqrFsqqxNWSXRYnRT65rzoAb4nALnqZdOZk/cpe7Ki5F2woz8
Tci0czQaZx4+kMGaXoAkZIcptS9oAm9BpCcX8J0tD6/U/rHW59JGB+xw6Cmq
rb9v1TkxQUPZcvzxN3qO4r4l3X6weRqO5U3tJdPh/qqpt23faJbY1c9hjJiz
GeD1COYewhijrObSRsfEyTeT21OwAf++x/Xy+4SLyx9t67zlMxxvNdN5nAvm
Xrlwhud93qLgWdxrzl/uLj50WbzS0bSJAM+Zk/+94BppjieAlpyOrz2aRuE1
T/F3/gEqRHWS50SHw043x9AavX1W4bF4uZhLOcV6xWmNckXWk7RYxw6R/8LW
Ygx0CJy+wMHw9lLR49Hf+ST5Nup88+TjnVK2YcYGjN/D+D6M8V3Qh554loT7
vOX9RWbchTFfayzfMFw2tb3whEzpOWXm34ExXy31/8ZY+QcrdmtC\
\>"], "Output",
 Evaluatable->False,
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{176, 93},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[47]//TableForm=",
 CellID->276177033]
}, Open  ]],

Cell[TextData[{
 "Timings for the integral, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "1"], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox["x", "10"], "+", 
        FractionBox["1", 
         SqrtBox["y"]]}], ")"}], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", singular along ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " axis with singularity handlers ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 ", ",
 Cell[BoxData["\"\<DuffyCoordinates\>\""], "InlineFormula"],
 ", and ",
 Cell[BoxData["IMT"], "InlineFormula"],
 " and with no singularity handling."
}], "MathCaption",
 CellID->479551743],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{
    RowBox[{
     RowBox[{"{", 
      RowBox[{"#", ",", 
       RowBox[{
        RowBox[{"Timing", "[", 
         RowBox[{"NIntegrate", "[", 
          RowBox[{
           RowBox[{
            FractionBox["1", 
             SqrtBox["y"]], "+", 
            SuperscriptBox["x", "10"]}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
           RowBox[{"Method", "\[Rule]", 
            RowBox[{"{", 
             RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
              RowBox[{"\"\<SingularityHandler\>\"", "\[Rule]", "#"}]}], 
             "}"}]}], ",", 
           RowBox[{"MaxRecursion", "\[Rule]", "20"}]}], "]"}], "]"}], "[", 
        RowBox[{"[", "1", "]"}], "]"}]}], "}"}], "&"}], "/@", 
    RowBox[{"{", 
     RowBox[{
     "Automatic", ",", "\"\<DuffyCoordinates\>\"", ",", "\"\<IMT\>\"", ",", 
      "None"}], "}"}]}], ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"None", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"StyleForm", "[", 
         RowBox[{
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{"\"\<Singularity\>\"", ",", " ", "\"\<handler\>\""}], 
            "}"}], "]"}], ",", 
          RowBox[{"FontFamily", "\[Rule]", "Times"}]}], "]"}], ",", 
        RowBox[{"StyleForm", "[", 
         RowBox[{"\"\<Time (s)\>\"", ",", 
          RowBox[{"FontFamily", "\[Rule]", "Times"}]}], "]"}]}], "}"}]}], 
     "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[46]:=",
 CellID->515634090],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWt1S1DAULtufxQWVwVHBn5EncLx3uHJG/Bl9AhV3kBEuHBV4Ae9RLn0N
3jA2TdKefDlJk24XZGbb5jQ5Ob9fTlLezc+ODr/Nz44P5jt7J/MfR8cHpzsv
v5/UpHwly7LL+vdxK6ufRf3U/C6FqO/uNa9fZnnzmJlHSZvoe1Z07xS1qO/7
+q18OanvJWnTew7tknDveIvmb2LoZUvSUweoeUvNPrO9cs25SOXsUEuYj+co
55MuWFHtqtWwvY6v4w2WWnHyJrbH1UQSPy2qQUF7Fdrihe49VN429jruJi3I
/B8SZDeZQTMh1lY0B+3sBBlJv0lm9zTz7qfam4/7BMmpxTwakB4/E+SLmzPV
0q68Lu6damuuCRfnTAxydBWLPux1rxYSt8Y3f0ZYg81fBY/CkklJpAqjMGsD
Np5pr0OvPa8PbcfUhI9AnGHKSelHrCSJ2+tCEuPYypZI2BjKW+yqZEWfm2hK
yWhTofhjm0PKoRL3rFBmoh5KGZa1wwCqucXhS4LMyfXHaNjPI9qsvkub+/CV
uxsEi8HXx0JFxEXTegJS/mmoheY5aQPCiFdmXe2CkcKhPuJvCdEWRnVEbRPJ
8r4ueLTvq7mn+t7NG/AZQ0OtjZ1Ss3Loyo3zJ88wBaogueuMmHn5LHFdA+uS
KjbvRPbUp0l1P2f1FJ2uyGM4A2mvcu0UtA5W+H17okon8UhescZhZXxdfjG4
w/Gk+LS2uOUH1SFjWD+AXQt4oEUIQFTO7oEd50I7MB6/lrNHtWUPe0clDtWc
rqz8ikh5VcJd5bCSa2nWuZKvN7da071biiV749tPwXwkSK0t6qzaazAO44uX
/CH0ugft7QR9l1VD4lqlLCCNdlPY/p9pLz0QfN49FWkVZaX5pO7Y3bMCbo+B
O3HZZ5q5NR+2dwVf85mYpXmE2lBrxCCnkTtUMzXvVukoeg0gRdxuwen13Ds2
tRrj6qxQDqF8tgUbKcIaJ51IdRaO44jS48q1y3KJO9NA3n3nyPREQ1kJcSwu
AlKs5KtRxrRLXy1IZTD7x6u2wwLREplr8Sc1Qyv+frvRqoFWD863KoJvFHEp
QpbaWynaYgZgVKVoa3+lQLvNoL0N7TcJUsfh4OJrducz6ZtnwP+F8K/D3CmG
8RezDsvJhqzJQq21NmWVxIvRTY5T50HqYNdZ5jyrppPjlTvUHXmxpJkwN38T
Mq0ijcYTDx/AIU0vQBIyw5TaFzSBHRGp0DP4ApeHR2r/WONzYeMD1jn0FNXW
3zfqnJhAUTYdf/yN7iO5bwq3KlRvw7G8ob1k6txfDfW27RvNEiv8Q2gj2mwE
eD2CvgfQxihruLTRUTn5ZnJ7Cjbgd39cRb9HuLj80bbOjp/heEt153EumHtl
bw/P3t6i4G7nNecvdxYfuvSPdDRVEeA5i/LvDv4jzfE80JLT8bVH0yi85in+
XUCAClGd5Lmsw2GnrmNoSm+fVXgsHhZzKSdarzitUa7I9SQt1rFG5L+wtRgD
FQKnL3AwvL1U9Hj0dz7dz/2C257dudhGKFvQYx3a76F9H9r4rduHnniuhPO8
5f1FetyFNr/WWL5huGxoe+FpmdRzyvS/A21+tdT/p7HyD/RZcTw=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{176, 93},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[46]//TableForm=",
 CellID->106094206]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Cauchy principal value integration", "Subsection",
 CellTags->"CauchyPrincipalValue",
 CellID->295594003],

Cell[TextData[{
 "To evaluate the Cauchy principal value of an integral, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " uses the strategy ",
 Cell[BoxData[
  ButtonBox["PrincipalValue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrincipalValue"]], "InlineFormula"],
 ". "
}], "Text",
 CellTags->{"S5.79.1", "1.2"},
 CellID->41257904],

Cell["Cauchy principal value integration with singular point at 2.", \
"MathCaption",
 CellID->169961519],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox[
    SqrtBox["x"], 
    RowBox[{"x", "-", "2"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2", ",", "5"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<PrincipalValue\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[153]:=",
 CellID->25600909],

Cell[BoxData["2.363548753370899`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[153]=",
 CellID->68093198]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s",
 StyleBox[" ", "SubsubsectionIcon"],
 Cell[BoxData[
  ButtonBox["PrincipalValue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrincipalValue"]], "InlineFormula"],
 " uses the strategy specified by its ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option to work directly on those regions where there is no difficulty and \
by pairing values symmetrically about the specified singularities in order to \
take advantage of the cancellation of the positive and negative values. "
}], "Text",
 CellTags->{"S5.79.1", "1.3"},
 CellID->208005087],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
method specification used to compute estimates over subregions\
\>", "TableText"]},
   {"SingularPointIntegrationRadius", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "a number ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["\[Epsilon]", "TR"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " or a list of numbers ",
     Cell[BoxData[
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["\[Epsilon]", "TR"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["\[Epsilon]", "TR"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"], ",", 
        SubscriptBox[
         StyleBox["\[Epsilon]", "TR"], 
         StyleBox["n", "TI"]]}], "}"}]], "InlineFormula"],
     " that correspond to the singular points ",
     Cell[BoxData[
      RowBox[{
       SubscriptBox[
        StyleBox["b", "TI"], 
        StyleBox["1", "TR"]], ",", 
       SubscriptBox[
        StyleBox["b", "TI"], 
        StyleBox["2", "TR"]], ",", 
       StyleBox["\[Ellipsis]", "TR"], ",", 
       SubscriptBox[
        StyleBox["b", "TI"], 
        StyleBox["n", "TI"]]}]], "InlineFormula"],
     " in the range specification; with each pair ",
     Cell[BoxData[
      RowBox[{"(", 
       RowBox[{
        SubscriptBox[
         StyleBox["b", "TI"], 
         StyleBox["i", "TI"]], ",", 
        SubscriptBox[
         StyleBox["\[Epsilon]", "TR"], 
         StyleBox["i", "TI"]]}], ")"}]], "InlineFormula"],
     " is formed an integral of the form\n",
     Cell[BoxData[
      RowBox[{
       SubsuperscriptBox["\[Integral]", "0", 
        StyleBox["\[Epsilon]", "TR"]], 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{
           StyleBox["f", "TI"], 
           RowBox[{"(", 
            RowBox[{
             StyleBox["b", "TI"], "+", 
             StyleBox["t", "TI"]}], ")"}]}], "+", 
          RowBox[{
           StyleBox["f", "TI"], 
           RowBox[{"(", 
            RowBox[{
             StyleBox["b", "TI"], "-", 
             StyleBox["t", "TI"]}], ")"}]}]}], ")"}], 
        RowBox[{"\[DifferentialD]", 
         StyleBox["t", "TI"]}]}]}]], "InlineFormula",
      UnderoverscriptBoxOptions->{LimitsPositioning->True}]
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->355362523],

Cell[TextData[{
 Cell[BoxData["\"\<PrincipalValue\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->43334287],

Cell["Thus the specification", "Text",
 CellTags->{"S5.79.1", "1.3"},
 CellID->401707095],

Cell[BoxData[
 RowBox[{
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"], "[", 
  RowBox[{
   RowBox[{
    StyleBox["f", "TI"], "[", 
    StyleBox["x", "TI"], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     StyleBox["x", "TI"], ",", 
     StyleBox["a", "TI"], ",", 
     StyleBox["b", "TI"], ",", 
     StyleBox["c", "TI"]}], "}"}], ",", 
   RowBox[{
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], "->", 
    RowBox[{"{", 
     RowBox[{"\"\<PrincipalValue\>\"", ",", 
      RowBox[{
       ButtonBox["Method",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Method"], "->", 
       StyleBox["methodspec", "TI"]}], ",", 
      RowBox[{"\"\<SingularPointIntegrationRadius\>\"", "->", 
       StyleBox["\[Epsilon]", "TR"]}]}], "}"}]}]}], "]"}]], "DisplayMath",
 CellTags->{"S5.79.1", "1.3"},
 CellID->585477522],

Cell["is evaluated as", "Text",
 CellID->266656539],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", 
      RowBox[{"b", "-", "\[Epsilon]"}]], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}], "+", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "\[Epsilon]"], 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        RowBox[{"f", "(", 
         RowBox[{"b", "+", "t"}], ")"}], "+", 
        RowBox[{"f", "(", 
         RowBox[{"b", "-", "t"}], ")"}]}], ")"}], 
      RowBox[{"\[DifferentialD]", "t"}]}]}], "+", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"b", "+", "\[Epsilon]"}], "c"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 UnderoverscriptBoxOptions->{LimitsPositioning->True},
 CellTags->{"S5.79.1", "1.4"},
 CellID->453629078],

Cell[TextData[{
 "where each of the integrals is evaluated using ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " with ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Method",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Method"], "->", 
   StyleBox["methodspec", "TI"]}]], "InlineFormula"],
 ". If ",
 Cell[BoxData[
  FormBox["\[Epsilon]", TraditionalForm]], "InlineFormula",
  GridBoxOptions->{
  GridBoxItemSize->{
   "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
    "RowsIndexed" -> {}}}],
 " is not given explicitly, a value is chosen based upon the differences ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["b", "TI"], "-", 
    StyleBox["a", "TI"]}], TraditionalForm]], "InlineFormula",
  GridBoxOptions->{
  GridBoxItemSize->{
   "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
    "RowsIndexed" -> {}}}],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["c", "TI"], "-", 
    StyleBox["b", "TI"]}], TraditionalForm]], "InlineFormula",
  GridBoxOptions->{
  GridBoxItemSize->{
   "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
    "RowsIndexed" -> {}}}],
 ". The option ",
 Cell[BoxData["SingularPointIntegrationRadius"], "InlineFormula"],
 " can take a list of numbers that equals the number of singular points. For \
the derivation of the formula see [",
 ButtonBox["DavRab84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#587039710"],
 "]."
}], "Text",
 CellTags->{"S5.79.1", "1.5"},
 CellID->650257369],

Cell[TextData[{
 "This finds the Cauchy principal value of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{
      RowBox[{"-", "1"}], "/", "2"}], "1"], 
    RowBox[{
     FractionBox["1", 
      RowBox[{"x", "+", 
       SuperscriptBox["x", "2"]}]], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellTags->{"S5.79.1", "1.7"},
 CellID->971403751],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    RowBox[{"x", "+", 
     SuperscriptBox["x", "2"]}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", 
      FractionBox["1", "2"]}], ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "PrincipalValue"}]}], "]"}]], "Input",
 CellTags->"S5.79.1",
 CellLabel->"In[14]:=",
 CellID->244484095],

Cell[BoxData[
 RowBox[{"-", "0.6931471805596523"}]], "Output",
 ImageSize->{138, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"S5.79.1",
 CellLabel->"Out[14]=",
 CellID->214486737]
}, Open  ]],

Cell[TextData[{
 "Here is the Cauchy principal value of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "2"}], "1"], 
    RowBox[{
     FractionBox["1", 
      RowBox[{"x", "+", 
       SuperscriptBox["x", "2"]}]], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ". Note that there are two singularities that need to be specified. "
}], "MathCaption",
 CellTags->{"S5.79.1", "1.8"},
 CellID->412596090],

Cell[CellGroupData[{

Cell["\<\
NIntegrate[1/(x+x^2), {x, -2, -1, 0, 1},Method->PrincipalValue]\
\>", "Input",
 CellTags->"S5.79.1",
 CellLabel->"In[114]:=",
 CellID->5880689],

Cell[BoxData[
 RowBox[{"-", "1.386294361119131`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"S5.79.1",
 CellLabel->"Out[114]=",
 CellID->595233216]
}, Open  ]],

Cell[TextData[{
 "The singular points can be specified using the ",
 Cell[BoxData[
  ButtonBox["Exclusions",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Exclusions"]], "InlineFormula"],
 " option."
}], "MathCaption",
 CellTags->{"S5.79.1", "1.8"},
 CellID->517376587],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"(", 
     RowBox[{"x", "+", 
      RowBox[{"x", "^", "2"}]}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", "PrincipalValue"}], ",", 
   RowBox[{"Exclusions", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "0"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[30]:=",
 CellID->255508626],

Cell[BoxData[
 RowBox[{"-", "1.386294361119131`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->337439646]
}, Open  ]],

Cell["\<\
This checks the value. The result would be 0 if everything were done exactly. \
\
\>", "MathCaption",
 CellTags->{"S5.79.1", "1.9"},
 CellID->21857125],

Cell[CellGroupData[{

Cell["% + 2Log[2]", "Input",
 CellTags->"S5.79.1",
 CellLabel->"In[31]:=",
 CellID->175583453],

Cell[BoxData["7.596145934485321`*^-13"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"S5.79.1",
 CellLabel->"Out[31]=",
 CellID->79835425]
}, Open  ]],

Cell[TextData[{
 "It should be noted that the singularities must be located exactly. Since \
the algorithm pairs together the points on both sides of the singularity, if \
the singularity is slightly mislocated the cancellation will not be \
sufficiently good near the pole and the result can be significantly in error \
if ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " converges at all. "
}], "Text",
 CellTags->{"S5.79.1", "1.10"},
 CellID->732146786],

Cell[CellGroupData[{

Cell["Sampling points visualization", "Subsubsection",
 CellID->420809027],

Cell["Consider the calculation of the principal value of ", "Text",
 CellID->44159629],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubsuperscriptBox["\[Integral]", "0", "2"], 
   RowBox[{
    FractionBox["1", 
     RowBox[{"log", "(", "x", ")"}]], 
    RowBox[{
     RowBox[{"\[DifferentialD]", "x"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->181243959],

Cell["\<\
The following examples show two ways of visualizing the sampling points. The \
first shows the sampling points used. Since the integrand is modified in \
order to do the principal value integration, it might be desired to see the \
points at which the original integrand is evaluated. This is shown on the \
second example.\
\>", "Text",
 CellID->86677025],

Cell[TextData[{
 "Here are sampling points used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ". There are no points over the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     FractionBox["3", "4"], ",", 
     FractionBox["5", "4"]}], "]"}], TraditionalForm]], "InlineMath"],
 ", because of ",
 Cell[BoxData[
  ButtonBox["PrincipalValue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrincipalValue"]], "InlineFormula"],
 "'s",
 " integration ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{
        SubsuperscriptBox["\[Integral]", "0", 
         RowBox[{"1", "-", 
          RowBox[{"1", "/", "4"}]}]], 
        RowBox[{
         FractionBox["1", 
          RowBox[{"log", "(", "x", ")"}]], 
         RowBox[{"\[DifferentialD]", "x"}]}]}], "+", 
       RowBox[{
        SubsuperscriptBox["\[Integral]", "0", 
         RowBox[{"1", "/", "4"}]], 
        RowBox[{"(", 
         RowBox[{
          FractionBox["1", 
           RowBox[{"log", "(", 
            RowBox[{"1", "+", "t"}], ")"}]], "+", 
          FractionBox["1", 
           RowBox[{"log", "(", 
            RowBox[{"1", "-", "t"}], ")"}]]}], ")"}]}]}], ")"}], 
     RowBox[{"\[DifferentialD]", "t"}]}], "+", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"1", "+", 
       RowBox[{"1", "/", "4"}]}], "2"], 
     RowBox[{
      FractionBox["1", 
       RowBox[{"log", "(", "x", ")"}]], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", and there are sampling points over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", 
     FractionBox["1", "4"]}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->427238965],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"k", "=", "0"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"1", "/", 
        RowBox[{"Log", "[", "x", "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1", ",", "2"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<PrincipalValue\>\"", ",", 
          RowBox[{"\"\<SingularPointIntegrationRadius\>\"", "\[Rule]", 
           RowBox[{"1", "/", "4"}]}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", 
           RowBox[{"++", "k"}]}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"tbl", ",", 
   RowBox[{"PlotRange", "->", "All"}]}], "]"}]}], "Input",
 CellLabel->"In[154]:=",
 CellID->323299145],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlglQVWUYhk8uY1OgODaJS3ZtQS1MExAX1E9ZBFyAyw73wl21HFNcBoTJ
8YwNqK2So4KFAaOgTdiOaGlHQL0YplIpadktSSYrB4GMUiT0PsfGmGGYd17e
+73f8//nwGjbcqOzj6IoVb3ft38aNuaPaFmxZpZy56tt1uYyr8Mr+5ajFcny
HTW1uHg/+n7Je6G4b/SYY2gfUYc02juizqAfEs/PZrSvqL8HvO1VfRE9Usp3
DVs3PvtntEHaQ75vbvT6Bf2YtAY9+o3rx8voJ6St0xU88lQr2k88fZ9DjxVP
3zfRT9G3Eu1P32r0M/Q9gp5IXxf6Wfp+hZ5E3yZ0AH2/RQfS9yw6iL7n0JNl
TV7LiNdb9f7B0rlj0JKW5bqeIrmjli1dVKDvO1Ximl794uHpOo9pYrjhNeal
qkvo6fTV+YWIu19H84tGN3qGZKRfVps2/YCeKQPrHh8T0HEePUuGnXA/HWz4
Di3i82Cw69JEzksV+Kbgz4bvevzZ8N2OPwe+Ffhz4Pshfih9a/BD4XsYPwy+
tfhh8D2KHw5f7psaDt/j+BHwhbcaAV+d/1z4cj7qXPjq5xcJ36/xI+mrn3cU
fE/jR8H3JH40fE/gR8O3AX8efLlf6jz4RuLPh+9K/Pnw3YC/AL7b8BfAtwx/
IX334C+E7z78GPh+hB8D30/xY+HL8yGx8OV5V2PhC28tFr46/zj4cj4SB1/O
T42Dbx35OPpy3ooRvhp5I3wPkTfC9yB5I3wPkI+HL/dL4uE7hXw8fG3k4+Gb
Qz4BvgXkE+BbSD6BvkXkE+C7k3wifHlfSiJ8d5NPhC/Ph5YIX/39lARfeEsS
fHX+SfDlfLQk+Ornlwzfj8kn01c/72T4fkA+Gb5V5FPg+y75FPjuJZ8CX+6X
lnIvDyVVzDdLqkxDd5FPFdf5a0E7lrCfmioxvvk7zF1oLVX2JL0RdbVH3z9N
uipru3t0LWmyvOf2l55PY54+P02GPuKzZXj4OvLp0r+yfvaB94zk0yW+budv
WzcNJJ/eyyd5b1Z12UxPPl1uDekM0a5cm+HJm+Rgfnnq4tFNIZ68SQomVPQf
aeua5smbJLXe+1RDQSnzTeJX419SFMt5KmY5s3py5MRmvb9ZXjbNOffA3f5m
2ZDdknl3H80sNwq7/9tPybh3P/mfVjPkrauuuuaW18hnSPnawPHZ15eQz5Sc
7UdbLrQGkM+UqM+WfTJ090XPvmqm3JzkP2FF3wT2zxRtQKCvX9Yq9rdI0RW/
+9wH+3j2N1jkzuPj28nnWcTL2bZ1xGa0xSIbfircUpz7J/0s8v74VfJr/XWP
LrVw37roa2GffzzabeG+3qC/lfvc7dEGK/f9FvOtPA89zLfyvKBVq9TXerdP
8aJfqVUKJnWd8I5Ea71+m6uh8XO02yrN26aWrrbRX7FJ1vrcpTPH0d9go+9f
zLfJ5rUFW46cRVtsMvb5Y5mvlLCfapOQQcGTA0L/Zr5N8k4OCGrfj9Z6/Rnt
3sc70G5d01+x8/tog53PQ4udefCw2OkDL9VOX3SpnX1uMt/OvvB22+HBeSgO
eKENDniixSFa2r7Ff8yzML9X+y/qLuq3kfkOkS+brmfnvMN8h3QVNvar0d8v
mkOU4T75Yd68z9wO+vK+V5yidOQcMufWM98p6y803ios4e+lOKWo4snRgwc3
Mt8pgSXjsqrX8f+Z6pStEdbo04dPMd8p/wLH7BEJ
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{364, 224},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[156]=",
 CellID->9726718]
}, Open  ]],

Cell["\<\
This defines a function which accumulates the argument values given to the \
integrand.\
\>", "MathCaption",
 CellID->31270416],

Cell[BoxData[
 RowBox[{
  RowBox[{"Clear", "[", "f", "]"}], ";", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x_", "?", "NumericQ"}], "]"}], ":=", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"AppendTo", "[", 
      RowBox[{"tbl", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"++", "k"}]}], "}"}]}], "]"}], ";", 
     RowBox[{"1", "/", 
      RowBox[{"Log", "[", "x", "]"}]}]}], ")"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->221567852],

Cell[TextData[{
 "Here are the points at which the integrand has been evaluated. Note the \
symmetric pattern over the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     FractionBox["3", "4"], ",", 
     FractionBox["5", "4"]}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->305109895],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"tbl", "=", 
   RowBox[{"{", "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1", ",", "2"}], "}"}], ",", 
    RowBox[{"Method", "->", 
     RowBox[{"{", 
      RowBox[{"\"\<PrincipalValue\>\"", ",", 
       RowBox[{"\"\<SingularPointIntegrationRadius\>\"", "\[Rule]", 
        RowBox[{"1", "/", "4"}]}]}], "}"}]}]}], "]"}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{"tbl", ",", 
   RowBox[{"PlotRange", "->", "All"}]}], "]"}]}], "Input",
 CellLabel->"In[166]:=",
 CellID->644335427],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlAlwjHcYxj/COJqQoVNRqks10ZIKkcQR8roTSch9H5u9UkYljnFNjW90
RPUSNUhcXRmCTmkrmjhaXRIkJqhQgtJVqZQ6IolIJaRhf5+O7szOzjPPPvs+
7+///7avISPC3FZRlP2t72efuo+X96qavTBAef6qCcje6nx4jlMeWpFMtz4j
cnOL0B1l8Qe5TlM8jqNdRe1ebqwLOot+VRyflWg3Ue94b3IuvIbuLXnbei71
nP8HWie1/r9Vljv/ie4n1T5vni/9/Sa6v9TUl/r1PlONdhdH3/fRA8TR90v0
u/TdgR5E30L0e/Q9gvaibyl6CH1Po4fStwLtTd9f0cPoewHtQ9+LaF9ZuLiq
1xfVWn8/qd/QdUZVhqaHy6I+s2ZasrR9R0h4xWc/vzZK4zFSdE3OHh/tvoEe
RV+Nn7/Y29VVfhhhR4+WlMSbasXKq+gx0qX4LQ/vusvoAOl50j7QT3cJLeL6
il/pDS/OSxX4xuGPhe8y/LHwXY8/Dr75+OPg+z3+ePruxx8P38P4E+B7FH8C
fI/hT4Qv902dCN8T+JPgC291Enw1/pPhy/mok+GrnV8gfM/hB9JXO+8g+P6C
HwTfU/hT4HsSfwp8y/CD4cv9UoPhG4gfAt85+CHwXYEfCt91+KHw3Yo/lb47
8afCdw/+NPjuxZ8G3x/ww+DL8yFh8OV5V8PgC29bGHw1/uHw5XwkHL6cnxoO
32Ly4fTlvJUI+NrIR8D3J/IR8D1IPgK+B8hHwpf7JZHwHU4+Er4G8pHwXUA+
Cr5Z5KPgu5p8FH1zyEfBdwv5aPjyfynR8N1OPhq+PB+2aPhq/08x8IW3xMBX
4x8DX87HFgNf7fxi4VtAPpa+2nnHwvc78rHw3U0+Dr5fk4+D7y7ycfDlftni
XuahxEty8+bdST22kY+X0ssPfDbMYD81Xqa5Ld+Q3Ii2xcvOmFVB91q0/ROk
ccfRJy2algTJaHn20vIJzNPmJ0iPN1zXvD5xKflEab+jZOyBbyLIJ0pk8Za/
167sQj6xlU/srszCrWMc+UR52r3e33b7wWhHPkkOLs+LT+9b4e/IJ0nW4Pz2
vQ2NIx35JIkvcTlTlmVlfpK47x+0OSeM81SS5ew830CvSq1/snySNO5i5xf9
k2XF/KrUF/vYkqVp9ZP/9lNSXt5P/qfVFNl4r7S4supz8imSt2SY5/yGGeRT
ZcH6Y1VXqr3Jp0rQoVn7emy/5thXTZXmoYMGz3aKYv9UsXUY5uaeOZf99ZJz
272N/WBbx/46vTx/fNzq+T29OJtr1vbKRuv1suL66jW5ix7STy/fes6VWyUN
Dm3Vc98a6atnn8cObddzX5von8Z9fuLQujTu+1Pmp/E8tDA/jecFraZJyVGX
2uHO9LOmSdbQxpMugWhbq19TWlb+I9qeJpXrRljnGeivGCRz2aKZY96hv85A
30fMN0j2kqw1Ry6g9QYZMP146qeb2U81iH9XP1/v8f8w3yCLT3XwqS1C21r9
0bUuJ+rQdk3TXzHyfbTOyO+hxcg8eOiN9IGXaqQv2mpkn2bmG9kX3nYjPDgP
xQQvtM4ET7SYZHt4Trd99+8z3ySeFwsa3u5Yw3yThKZ33nT3K3yrSRLKHw/p
5IVvM0n+9KDACe3w7a354CtXri7BV8xi9y1YavK4x3yzZJvcbaF38MUstubk
jNg+d5lvFmub8MDzmQ+Yb2b/O8zXdC3zzSJOedk7r99mvln0T/cUDVyFr1hE
Vxaibqy8xXyLZG68FLC3ex3zLZKwrrDoUNNfzLfIuX39+/dbhq9apCDnobGb
Ht9qkXzv9qcbzuDbLJK4J/1usCu+vTU/IKTT5Uf4Srr8C7CdS2E=
     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{184, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[168]=",
 CellID->26227620]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Double-Exponential Strategy", "Section",
 CellTags->{"DEstrategy", "DoubleExponential"},
 CellID->526196975],

Cell[TextData[{
 "The double-exponential quadrature consists of basically applying the \
trapezoidal rule after a variable transformation. The double-exponential \
quadrature was proposed by Mori and Takashi in 1974 and it was inspired by \
the so-called ",
 ButtonBox["IMT rule",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#150310618"],
 " and TANH rule. The transformation is given the name \"",
 "double-exponential",
 "\" since its derivative decreases double-exponentially when the integrand's \
variable reaches the ends of the integration region."
}], "Text",
 CellID->62223081],

Cell[TextData[{
 "The ",
 "double-exponential",
 " algorithm for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " is specified with the ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option value ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->215953538],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "->", "\"\<DoubleExponential\>\""}]}], "]"}]], "Input",
 CellLabel->"In[169]:=",
 CellID->74782051],

Cell[BoxData["2.000000000000003`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[169]=",
 CellID->204398443]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<ExtraPrecision\>\"", "50", Cell[
    "maximum extra precision to be used internally", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->190241735],

Cell[TextData[{
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->8402317],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 "double-exponential",
 " strategy can be used for one-dimensional and multidimensional integration. \
When applied to multidimensional integrals it uses the Cartesian product of \
the trapezoidal rule."
}], "Text",
 CellID->382257924],

Cell[TextData[{
 "A ",
 "double-exponential",
 " transformation ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Phi]", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
 " transforms the integral"
}], "Text",
 CellID->147235414],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubsuperscriptBox["\[Integral]", "a", "b"], 
   RowBox[{
    RowBox[{"f", "(", "t", ")"}], 
    RowBox[{"\[DifferentialD]", "x"}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialStrategy1",
 CellID->598790360],

Cell["into", "Text",
 CellID->333021865],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "\[Infinity]"}], 
     RowBox[{"+", "\[Infinity]"}]], 
    RowBox[{
     RowBox[{
      RowBox[{"f", "(", 
       RowBox[{"\[Phi]", "(", "t", ")"}], ")"}], " ", 
      RowBox[{
       SuperscriptBox["\[Phi]", "\[Prime]",
        MultilineFunction->None], "(", "t", ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialStrategy2",
 CellID->273635301],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"a", ",", "b"}], ")"}], TraditionalForm]], "InlineMath"],
 " can be finite, half-infinite (",
 Cell[BoxData[
  FormBox[
   RowBox[{"b", "=", "\[Infinity]"}], TraditionalForm]], "InlineMath"],
 "), or infinite (",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"a", "=", 
     RowBox[{"-", "\[Infinity]"}]}], ",", 
    RowBox[{"b", "=", "\[Infinity]"}]}], TraditionalForm]], "InlineMath"],
 "). The integrand ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " must be analytic in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"a", ",", "b"}], ")"}], TraditionalForm]], "InlineMath"],
 " and might have singularity at one or both of the end points."
}], "Text",
 CellID->380878990],

Cell[TextData[{
 "The transformed integrand decreases double-exponentially, that is, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"f", "(", 
      RowBox[{"\[Phi]", "(", "t", ")"}], ")"}], 
     RowBox[{"\[Phi]", "'"}], 
     RowBox[{"(", "t", ")"}]}], " ", "\[TildeTilde]"}], TraditionalForm]], 
  "InlineMath"],
 " ",
 Cell[BoxData[
  FormBox[
   RowBox[{"exp", "(", 
    RowBox[{
     RowBox[{"-", "c"}], " ", 
     RowBox[{"exp", "(", 
      RowBox[{"|", "t", "|"}], ")"}]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " as ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"|", "t", "|"}], "\[Rule]", 
    RowBox[{"\[PlusMinus]", "\[Infinity]"}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "Text",
 CellID->56205143],

Cell[TextData[{
 "The function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Phi]", "(", "t", ")"}], TraditionalForm]], "InlineMath"],
 " is analytic in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], ")"}], 
   TraditionalForm]], "InlineMath"],
 ". It is known that for an integral like (",
 
 CounterBox["NumberedEquation", "Double-exponentialStrategy2"],
 ") of an analytic integrand the trapezoidal rule is an optimal rule [",
 ButtonBox["Mori74",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#137456387"],
 "]."
}], "Text",
 CellID->46297055],

Cell["\<\
The transformations used for the different types of integration regions are:\
\>", "Text",
 CellID->257069525],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "b"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", 
       RowBox[{"x", "\[DoubleLongRightArrow]", "x"}]}]}]}], "=", 
    RowBox[{
     FractionBox[
      RowBox[{"a", "+", "b"}], "2"], "+", 
     RowBox[{
      FractionBox["1", "2"], " ", 
      RowBox[{"(", 
       RowBox[{"b", "-", "a"}], ")"}], " ", 
      RowBox[{"tanh", "(", 
       RowBox[{
        FractionBox["1", "2"], " ", "\[Pi]", " ", 
        RowBox[{"sinh", "(", "x", ")"}]}], ")"}]}]}]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialStrategy3",
 CellID->93539995],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "\[Infinity]"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", 
       RowBox[{"x", "\[DoubleLongRightArrow]", "x"}]}]}]}], "=", 
    RowBox[{"a", "+", 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{
       FractionBox["1", "2"], " ", "\[Pi]", " ", 
       RowBox[{"sinh", "(", "x", ")"}]}]]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->14552469],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"-", "\[Infinity]"}], "\[Infinity]"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", 
       RowBox[{"x", "\[DoubleLongRightArrow]", "x"}]}]}]}], "=", 
    RowBox[{"sinh", "(", 
     RowBox[{
      FractionBox["1", "2"], " ", "\[Pi]", " ", 
      RowBox[{"sinh", "(", "x", ")"}]}], ")"}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->55188097],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox["a", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"b", " "}], TraditionalForm]], "InlineMath"],
 "are finite numbers."
}], "Text",
 CellID->44340047],

Cell[TextData[{
 "The trapezoidal rule is applied to (",
 
 CounterBox["NumberedEquation", "Double-exponentialStrategy2"],
 "):"
}], "Text",
 CellID->505120836],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"DE", "(", "h", ")"}], "=", 
   RowBox[{"h", " ", 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"i", "=", 
       RowBox[{"-", "\[Infinity]"}]}], "\[Infinity]"], 
     RowBox[{
      RowBox[{"f", "(", 
       RowBox[{"\[Phi]", "(", 
        RowBox[{"i", " ", "h"}], ")"}], ")"}], " ", 
      RowBox[{
       SuperscriptBox["\[Phi]", "\[Prime]",
        MultilineFunction->None], "(", 
       RowBox[{"i", " ", "h"}], ")"}]}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialStrategy4",
 CellID->98724665],

Cell[TextData[{
 "The terms in (",
 
 CounterBox["NumberedEquation", "Double-exponentialStrategy4"],
 ") decay double-exponentially for large enough ",
 Cell[BoxData[
  FormBox[
   RowBox[{"|", "i", "|"}], TraditionalForm]], "InlineMath"],
 ". Therefore the summation in (",
 
 CounterBox["NumberedEquation", "Double-exponentialStrategy4"],
 ") is cut off at the terms that are too small to contribute to the total \
sum. (A criterion similar to (",
 
 CounterBox["NumberedEquation", "LocalAdaptiveStrategy1"],
 ") for the local adaptive strategy is used. See also the following",
 ButtonBox[" double-exponential example implementation",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#5698822"],
 ".)"
}], "Text",
 CellID->98189758],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s strategy ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " employs the ",
 "double-exponential",
 " quadrature."
}], "Text",
 CellID->308823815],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " strategy works best for analytic integrands; see ",
 ButtonBox["Comparison of double-exponential and Gaussian quadrature.",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#140889501"]
}], "Text",
 CellID->640131923],

Cell[TextData[{
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " uses the Cartesian product of double-exponential quadratures for \
multidimensional integrals."
}], "Text",
 CellID->798217516],

Cell["Cartesian double-exponential quadrature.", "MathCaption",
 CellID->2263677],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"x", "+", "y"}], "]"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<DoubleExponential\>\""}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "200"}]}], "]"}]], "Input",
 CellLabel->"In[48]:=",
 CellID->218973366],

Cell[BoxData["1.104569499663195`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[48]=",
 CellID->1316359695]
}, Open  ]],

Cell[TextData[{
 "As with the other Cartesian product rules, if ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " is used for dimensions higher than three, it might be very slow due to \
combinatorial explosion."
}], "Text",
 CellID->24165405],

Cell[TextData[{
 "The following plot illustrates the Cartesian product character of the ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " multidimensional integration."
}], "MathCaption",
 CellID->268211024],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"Sqrt", "[", "x", "]"}], 
        RowBox[{"Sqrt", "[", "y", "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", "\"\<DoubleExponential\>\""}], ",", 
       RowBox[{"MaxRecursion", "\[Rule]", "200"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"PointSize", "[", "0.005", "]"}], ",", 
     RowBox[{"Point", "/@", 
      RowBox[{"N", "[", "tbl", "]"}]}]}], "}"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[49]:=",
 CellID->25028486],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWtFOwyAUZdBui8Y4nQ8+7kU/RZ/8iGYx2R7MzNyXaPwD950V2rJ159JC
K51NSpNS4F7gcu7lFJq+JLvV61uyWy+TxfM2eV+tlx+Lp81WVokRY2wv7597
JvOpzGX3Pk3lk6aPMuVSQaiSUJqqlmXZ27IszmX5xVAqDlLGLbK5zKgqbpBl
+uxBpdFRwlgM1t0QjTH0dk00IlIzc9CZQ7+CaPAKm9Ei82iRox3d2jpxQ6/W
Ay42Kz8qP991aKkew2yhcLS9PZa0LzpetY0xjOvT73qtj4vnvHgiprNSvbbS
1E6X81Wb88kM+vpsXHsl0+9Dr0onKmbA9UwmwEV5SmI0gv45yPX8EHNs57ss
Svid+gn5ElkZZ2ji1wI4pkVjgiPlchsChKuhPIXyBZQvGyKE7aetrM6AmBpj
Jc2Z4IiNOTrbI4ZrqilCWDYiUjNewMIPFmpFfVmQCXjZYqdJrQviAdt22AbE
u0Fcv3MZM5+k0IaiVe0Zx6UNnjFwP2Bqg/jwYtauurb+6f60YiYlzPROEmMp
xJ/fVXze2lOLggfPjfjf+iOcZvWbL4bS30VcGBBPRHE6xEgzf11w8Y9FtxQD
aKvZk/2LYz+1w4upviDffSxU7d5U1nUdOe96HFoNL9Z8YSkasF6d3Wb245WW
9C+mA5OdMln/3ijtI2OYHuw/bzfZaynVf7Cn0Zk7RJ9tJ3QeVgmId4d4wDZ8
xfGHVz6Ivkx/uAzz+wQiQf9QGOa5J+AScPGPC+Wltu3c94v0z0I2+gUtiqZN
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{165, 164},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[50]=",
 CellID->251405315]
}, Open  ]],

Cell[TextData[{
 "Double-exponential quadrature can be used for singularity handling in \
adaptive strategies; see ",
 ButtonBox["Singularity handling",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#122144792"],
 "."
}], "Text",
 CellID->655737866],

Cell[CellGroupData[{

Cell["MinRecursion and MaxRecursion", "Subsection",
 CellID->168459464],

Cell[TextData[{
 "The option ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " has the same meaning and functionality as it does for ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " described in \"",
 ButtonBox["MinRecursion and MaxRecursion",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#280704764"],
 "\". ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " for ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " restricts how many times the trapezoidal quadrature estimates are \
improved; see ",
 ButtonBox["Example implementation of double-exponential quadrature",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#5698822"],
 "."
}], "Text",
 CellID->87421077]
}, Open  ]],

Cell[CellGroupData[{

Cell["Comparison of double-exponential and Gaussian quadrature", "Subsection",
 CellTags->"DEvsGauss",
 CellID->140889501],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " strategy works best for analytic integrands. For example, the following \
integral is done by ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " three times faster than the Gaussian quadrature (using a global adaptive \
algorithm)."
}], "Text",
 CellID->107867122],

Cell[TextData[{
 "Integration with ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 StyleBox[".", "Text"]
}], "MathCaption",
 CellID->440643516],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Log", "[", 
      FractionBox["1", "x"], "]"}], 
     SuperscriptBox["x", 
      RowBox[{"1", "/", "4"}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "10"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<DoubleExponential\>\"", ",", 
       RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
   "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[215]:=",
 CellID->303818724],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.001999000000012796`", ",", "1.7777777777777777`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[215]=",
 CellID->393420934]
}, Open  ]],

Cell[TextData[{
 "Integration with Gauss quadrature. (The default strategy of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 StyleBox[",", "Text"],
 " ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 StyleBox[",", "Text"],
 " uses by default a Gauss\[Dash]Kronrod integration rule with 5 Gaussian \
points and 6 Kronrod points.)"
}], "MathCaption",
 CellID->403397382],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Log", "[", 
      FractionBox["1", "x"], "]"}], 
     SuperscriptBox["x", 
      RowBox[{"1", "/", "4"}]]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "10"}], ",", 
    RowBox[{"MaxRecursion", "\[Rule]", "100"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
       RowBox[{"\"\<SingularityDepth\>\"", "\[Rule]", "\[Infinity]"}], ",", 
       RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
   "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[51]:=",
 CellID->596691180],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.008997999999998327`", ",", "1.7777777777691457`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[51]=",
 CellID->1585872896]
}, Open  ]],

Cell[TextData[{
 "Since ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " converges double-exponentially with respect to the number of evaluation \
points, increasing the precision goal increases a little the work done by ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 ". This is illustrated for two integrals, ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     FractionBox["1", 
      SqrtBox["x"]], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"20", " ", 
        RowBox[{"(", 
         RowBox[{"x", "-", "1"}], ")"}]}]], " ", 
      RowBox[{"sin", "(", 
       RowBox[{"256", " ", "x"}], ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ". Each table entry shows the error and number of evaluations."
}], "Text",
 CellID->308677690],

Cell[TextData[{
 "Double-exponential quadrature and Gaussian quadrature for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     FractionBox["1", 
      SqrtBox["x"]], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ". Increasing the precision goal does not change the number of sampling \
points used by ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->186554363],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"methods", "=", 
   RowBox[{"{", 
    RowBox[{"\"\<DoubleExponential\>\"", ",", "\"\<GlobalAdaptive\>\""}], 
    "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"pgoals", "=", 
   RowBox[{"Range", "[", 
    RowBox[{"5", ",", "15", ",", "2"}], "]"}]}], ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Outer", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{"k", "=", "0"}], ";", 
         RowBox[{"res", "=", 
          RowBox[{"NIntegrate", "[", 
           RowBox[{
            FractionBox["1", 
             RowBox[{"Sqrt", "[", "x", "]"}]], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"Method", "\[Rule]", "#1"}], ",", 
            RowBox[{"PrecisionGoal", "\[Rule]", "#2"}], ",", 
            RowBox[{"MaxRecursion", "\[Rule]", "20"}], ",", 
            RowBox[{"EvaluationMonitor", ":>", 
             RowBox[{"k", "++"}]}]}], "]"}]}], ";", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"Abs", "[", 
             RowBox[{"res", "-", "2"}], "]"}], "/", "2"}], ",", "k"}], 
          "}"}]}], ")"}], "&"}], ",", "methods", ",", "pgoals"}], "]"}], "//",
     "Transpose"}], ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"pgoals", ",", "methods"}], "}"}]}], ",", 
   RowBox[{"TableDepth", "\[Rule]", "2"}]}], "]"}]}], "Input",
 CellLabel->"In[217]:=",
 CellID->782846833],

Cell[BoxData[
 TagBox[
  TagBox[GridBox[{
     {
      StyleBox["\[Null]",
       ShowStringCharacters->False], 
      TagBox["\<\"DoubleExponential\"\>",
       HoldForm], 
      TagBox["\<\"GlobalAdaptive\"\>",
       HoldForm]},
     {
      TagBox["5",
       HoldForm], 
      RowBox[{"{", 
       RowBox[{"1.5543122344752192`*^-15", ",", "33"}], "}"}], 
      RowBox[{"{", 
       RowBox[{"1.5543122344752192`*^-15", ",", "132"}], "}"}]},
     {
      TagBox["7",
       HoldForm], 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "64"}], "}"}], 
      RowBox[{"{", 
       RowBox[{"1.5543122344752192`*^-15", ",", "132"}], "}"}]},
     {
      TagBox["9",
       HoldForm], 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "64"}], "}"}], 
      RowBox[{"{", 
       RowBox[{"8.881784197001252`*^-16", ",", "229"}], "}"}]},
     {
      TagBox["11",
       HoldForm], 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "64"}], "}"}], 
      RowBox[{"{", 
       RowBox[{"8.881784197001252`*^-16", ",", "273"}], "}"}]},
     {
      TagBox["13",
       HoldForm], 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "64"}], "}"}], 
      RowBox[{"{", 
       RowBox[{"8.881784197001252`*^-16", ",", "405"}], "}"}]},
     {
      TagBox["15",
       HoldForm], 
      RowBox[{"{", 
       RowBox[{"0.`", ",", "123"}], "}"}], 
      RowBox[{"{", 
       RowBox[{"8.881784197001252`*^-16", ",", "640"}], "}"}]}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxDividers->{
     "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}],
   {OutputFormsDump`HeadedRows, OutputFormsDump`HeadedColumns}],
  Function[BoxForm`e$, 
   MatrixForm[
   BoxForm`e$, 
    TableHeadings -> {{5, 7, 9, 11, 13, 15}, {
      "DoubleExponential", "GlobalAdaptive"}}, TableDepth -> 2]]]], "Output",
 ImageSize->{300, 139},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[219]=",
 CellID->131952507]
}, Open  ]],

Cell[TextData[{
 "Double-exponential quadrature and Gaussian quadrature for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     RowBox[{
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{"20", " ", 
        RowBox[{"(", 
         RowBox[{"x", "-", "1"}], ")"}]}]], " ", 
      RowBox[{"sin", "(", 
       RowBox[{"256", " ", "x"}], ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ". Increasing the precision goal does not change the number of sampling \
points used by ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 ". (The integrations are done without symbolic preprocessing.)"
}], "MathCaption",
 CellID->74266774],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"methods", "=", 
   RowBox[{"{", 
    RowBox[{"\"\<DoubleExponential\>\"", ",", "\"\<GlobalAdaptive\>\""}], 
    "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"pgoals", "=", 
   RowBox[{"Range", "[", 
    RowBox[{"6", ",", "10", ",", "2"}], "]"}]}], ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Outer", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{
         RowBox[{"k", "=", "0"}], ";", 
         RowBox[{"res", "=", 
          RowBox[{"NIntegrate", "[", 
           RowBox[{
            RowBox[{
             RowBox[{"Exp", "[", 
              RowBox[{"20", 
               RowBox[{"(", 
                RowBox[{"x", "-", "1"}], ")"}]}], "]"}], 
             RowBox[{"Sin", "[", 
              RowBox[{"256", " ", "x"}], "]"}]}], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"Method", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{"#1", ",", 
               RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], 
              "}"}]}], ",", 
            RowBox[{"PrecisionGoal", "\[Rule]", "#2"}], ",", 
            RowBox[{"MaxRecursion", "\[Rule]", "20"}], ",", 
            RowBox[{"EvaluationMonitor", ":>", 
             RowBox[{"k", "++"}]}]}], "]"}]}], ";", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{
            RowBox[{"Abs", "[", 
             RowBox[{"res", "-", "2"}], "]"}], "/", "2"}], ",", "k"}], 
          "}"}]}], ")"}], "&"}], ",", "methods", ",", "pgoals", ",", "1"}], 
     "]"}], "//", "Transpose"}], ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"pgoals", ",", "methods"}], "}"}]}], ",", 
   RowBox[{"TableDepth", "\[Rule]", "2"}]}], "]"}]}], "Input",
 CellLabel->"In[220]:=",
 CellID->887804540],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW8tu00AUnfgV2pKWChYggahQYccKWAB7WPERUYXULhCo9A/6sXxFNXgc
3/j6+I5nJrYTJ3KkPO547vvO44ydH8u765+/lnc3V8uL77fLP9c3V38vvv2+
zZvimVLqIX9/fa7y3zr/VbwftM6/V59PzWdsWouXSoH+APQroN85+l8CHYta
E6A/5h9RTr52SKN+aEVUaCmVFa+4JM333KHdcKdlfyM9LfkSyzfGDOU9EmLQ
xo9e9m3P3MJHdmViNGOpNV3TZaiRF3UjTT4lojRZR0irpN/oO/Lyxk8m5mss
Ps47S8ZKEmtgw8pALtR1NFC0UM+ZZEdS0U3/kH4zEk+eAJ04PZEqldMzPdzI
H29+I6Bd43tk+U24nESYH4acz3eVVcMVsqbuyu5TXa3hUZmxhDLnnNn3MXNo
x5k9Tp1X2q360DoSMy9/9jGfcutCbBVXztFmOcQzue9m/q6qwHzy3T2iFJxJ
qP9j3Y4GjA1p6QehCJpxYsEaCUuF6DnVzZk4btEnI8J/W209JNRLK+GpQ9uh
o9yMyQvhq58drFtYjxOHhLHgvwnjbiq5Wz1MyLZvTxZAu07wXDh3xtzvZ4yP
N6djRbO4Prn2t65aDVv9xpK/fcGtaDef702z63Sv/1V1wquBPqAdDS5xv7CX
OZwwqr1vV4wqza77ik2TUh/HHhM21UI9T9h0CGxKciguma7qdcKkLpljxaTo
Y7EaB92H677yThi0b09wzjJcmdAvEypjwpu790TKn2XnHORffWwW9qgJfw6d
PbjTrWDPWLviPtET76zvwOtDxJ54dsftUML4SuG6HYm01ceEPLfjmXiy0Dm7
NJaxdvYJZ9LdCf6UB3/qoxvONI/1DoE2P5ee3Jffb8VeX4C+B/qy4mo8jcSx
U7QukdUrZTGkmPHrSFNeIl096cTx1CqxivUuJKS8lUyPGS+3A58iPoFcI19i
4ZO1ElckcHGkyaVj5RifMjFPUT0DJF4e25xP2ca7C6m0YC1fO0JanZplPX5P
3KCvx/4R6RVXDjV3+8WG09L14mxFrrwtYG6xCmDF6PEMwK/eHCvWuCst3SQu
wXEYf1Wt48Cs7Lea7JH2iZ70ZO8x9hNkhWZR0iNmz8q9mY/tcWtZ7zrnZb36
uVC/yC1j0JjLbT9fCOvVsAGfjQiYcRgt/geoPVpBpyau8W/3r6ZzQ+9bkKtn
TfFdX2SJhFI0Z6DnOC5JIpfE9q66fq993YPvjFuu0U6VqjPV8lmPqwZqltd8
tJ0eoY+456f4+lpOFjT39THl3mMPfO7wU0bQ76EX6vFbs/HeQNvpiGQb4u2X
lc8NpHWu6yjUWGhOpz8JlpvrC12hYupPZ5n8Pwsk7wXIeVbYsvpfqZr9B3pw
mzE=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{253, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[222]=",
 CellID->302717391]
}, Open  ]],

Cell[TextData[{
 "On the other hand, for non-analytic integrands ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " is quite slow, and a global adaptive algorithm using Gaussian quadrature \
can resolve the singularities easily."
}], "Text",
 CellID->9192204],

Cell[TextData[{
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " needs more than 10000 integrand evaluations to compute this integral with \
a non-analytic integrand."
}], "MathCaption",
 CellID->333039582],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"k", "=", "0"}], ";"}], "\n", 
 RowBox[{"{", 
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     RowBox[{"Abs", "[", 
      RowBox[{"Sin", "[", 
       RowBox[{"3", "*", "x"}], "]"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}], ",", 
     RowBox[{"Method", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"\"\<DoubleExponential\>\"", ",", 
        RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}], 
     ",", 
     RowBox[{"MaxRecursion", "\[Rule]", "10"}], ",", 
     RowBox[{"EvaluationMonitor", ":>", 
      RowBox[{"k", "++"}]}]}], "]"}], ",", "k"}], "}"}]}], "Input",
 CellLabel->"In[52]:=",
 CellID->11930943],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.0000002315187455`", ",", "10185"}], "}"}]], "Output",
 ImageSize->{81, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[53]=",
 CellID->817068305]
}, Open  ]],

Cell["Gaussian quadrature is much faster for the integral.", "MathCaption",
 CellID->260812274],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"Abs", "[", 
       RowBox[{"Sin", "[", 
        RowBox[{"3", "*", "x"}], "]"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<GlobalAdaptive\>\"", ",", 
         RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}], 
      ",", 
      RowBox[{"MaxRecursion", "\[Rule]", "10"}], ",", 
      RowBox[{"EvaluationMonitor", ":>", 
       RowBox[{"k", "++"}]}]}], "]"}], ",", "k"}], "}"}]}]], "Input",
 CellLabel->"In[54]:=",
 CellID->2752420],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[54]:=",
 CellID->542235836],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1.999999949984792`", ",", "385"}], "}"}]], "Output",
 ImageSize->{65, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[54]=",
 CellID->839798625]
}, Open  ]],

Cell[TextData[{
 "Further, ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " might be slowed down by integrands that have nearly discontinuous \
derivatives, that is, integrands that are not \"very\" analytical."
}], "Text",
 CellID->34484985],

Cell["Here is an example with a not \"very\" analytical integrand.", \
"MathCaption",
 CellID->492435818],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{
      RowBox[{"16", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"x", "-", 
          FractionBox["\[Pi]", "4"]}], ")"}], "2"]}], "+", 
      FractionBox["1", "16"]}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<DoubleExponential\>\""}]}], "]"}], "//",
   "Timing"}]], "Input",
 CellLabel->"In[226]:=",
 CellID->450197686],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.01199800000000334`", ",", "2.7787844196279563`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[226]=",
 CellID->159217878]
}, Open  ]],

Cell["Again, Gaussian quadrature is much faster.", "MathCaption",
 CellID->35066339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     RowBox[{
      RowBox[{"16", " ", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"x", "-", 
          FractionBox["\[Pi]", "4"]}], ")"}], "2"]}], "+", 
      FractionBox["1", "16"]}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<GlobalAdaptive\>\""}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[227]:=",
 CellID->164538670],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.00599899999998757`", ",", "2.7787844196286824`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[227]=",
 CellID->404411912]
}, Open  ]],

Cell[TextData[{
 "Here are the plots of the integrand ",
 Cell[BoxData[
  FractionBox["1", 
   RowBox[{
    RowBox[{"16", " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"x", "-", 
        FractionBox["\[Pi]", "4"]}], ")"}], "2"]}], "+", 
    FractionBox["1", "16"]}]]]],
 " and its derivative."
}], "MathCaption",
 CellID->343915675],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Block", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"gr", ",", "gr1"}], "}"}], ",", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{"gr", "=", 
     RowBox[{"Plot", "[", 
      RowBox[{
       FractionBox["1", 
        RowBox[{
         RowBox[{"16", " ", 
          SuperscriptBox[
           RowBox[{"(", 
            RowBox[{"x", "-", 
             FractionBox["\[Pi]", "4"]}], ")"}], "2"]}], "+", 
         FractionBox["1", "16"]}]], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";", 
    RowBox[{"gr1", "=", 
     RowBox[{"Plot", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"D", "[", 
         RowBox[{
          FractionBox["1", 
           RowBox[{
            RowBox[{"16", " ", 
             SuperscriptBox[
              RowBox[{"(", 
               RowBox[{"x", "-", 
                FractionBox["\[Pi]", "4"]}], ")"}], "2"]}], "+", 
            FractionBox["1", "16"]}]], ",", "x"}], "]"}], "//", "Evaluate"}], 
       ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";", 
    "\[IndentingNewLine]", 
    RowBox[{"GraphicsArray", "[", 
     RowBox[{"{", 
      RowBox[{"gr", ",", "gr1"}], "}"}], "]"}]}]}], "\[IndentingNewLine]", 
  "]"}]], "Input",
 CellLabel->"In[228]:=",
 CellID->284519127],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWUtoXkUUnuY+/pjYNpa2VotN/tTHrtqFKIi48IUUEUFxIV3EEEkWWknT
Td3oRje60Y0FseALfCwqlopQQSyIBbEgFkSFUkHaRYtgcREojN+ZM49zZ+be
/H+Sgpfm3pm5c97fOWfu38dnlubnXphZWpidmXp0ceal+YXZg1OPHFjEUrFB
KXUcf/fuUBhrjMzfca3x7L5P4D6CzQXNCqIyl6I1XuRLvqd56dZqT8NvWviV
mbUKw5GM3OcNp82OzLMuk5WYmiQUQnMjxWqUSBLas41CSqRRlUieW7WO6Q5a
Kf9nGsr4O886KtZlUG2rDLdUh/XVtrQ4yOnYtn4tPdrlm5w+he7A7Ip6brNc
WTF3OZ51xOM+LetAHXlT2TUS0Esw0LOS3A5HfWPGHlk92uy9w0w++/Tnhflj
y8tX1Razf7/lHuN+OK77n/1k9rnPT313znKdWxO3B+4/8s3JP1595aTlttm+
faoZDRnHRoxcVtxu3ly48M8zT3985cryvseOMr9NeZQJffYJ7DgZpeDs1lsr
d0PLQg3RAw68+CUN2+p3r8UCor8Hzx3SK1WcFdIGBrdg4TMmrvADZeSoX/nq
xK+vv/YtpkfeOf3Rh2fyeVR3RYKy+cxPf+EfD0gGPRrrEEPrR9/7UU1gQE8j
Eg7EAIgEjDAALnGHHkwlNK6l3AHWRZy3aJE1rdWpDem5ftTeudo6Rdb3SO+3
3/qe3QRHDO97orgu0s1gblwi+vffLiFD2dlIKw63GovouBpSrVEKWahuWpW3
8liNa6n3Tge3tFIkOwhFVhauy5f/9cIHR6/gdzPdKtQegBWIhMPIGxOGDhEC
NuE6vL7WnnGRzXnEozGgCEpxdmH85hunulDkarOrKreYfbASGffy4a8xMF4q
wA995/z5v9XOdbTVVwsfMh60nUuzCAl2o+UA1jwFxLvs3h5x3mXmp3/4k41W
15s5OiLmNNypQ8eQz1o3O4o8DxT2Gc73SDfAxqnkaHteI5pNmZ1fHDuLyNGQ
sxL5BwCSZnss59vMOiAISx0WzD50S9q3yWO8cH4Q9k5F1FwzYCxMpuEeLXpw
AZZgHCKATYeWTpCUjUbKDRlcEnU/8uKomUMkBNPwzoYUFHxLz9WUkw53CBpv
lSPn/Uhb49mCYSHlse9QHp584gPtKh2dNJA5WILIsVZ5hbeL+6Onp0qJYkpe
ITl3u4rayLFUW+QUrAxc+OhEw7ustLR3ip5a+GmYcy/larjRzwE+COIBF0ft
C4wSvfPsLxeR6v7FStk96m3MfXnsjiLCPuAC5WzMfmNH814054ix4YgY4uAz
nVunSvsga+pO83Gtn440qyLNnaa7XOyx1WNVRG2rltnM9vORKuzmLKXhXrtv
kp6IUxntD9xSXUMexVR7LXImda5GwFlQP9X1VvMWEEZH0z7XcR7BqUTqyrUj
ZPWI1X2b4Tamm2fvGPXsEZQPFNsgJVTGyYg+rrQma2ofF+3aHUsfdyiw0vI9
M1gbrGM9Hn7wXSSwdhVyVX2ub/0xElm6dQgNU48FhPH7UDf6a9JU1R5HuDgO
w/iy8CgPeOR9MBvGr01Dg/TKRxsW0xlkGP1STyLACHN4z4fu9dDTo4lhNJSe
vi81sM29q6+bX3ZGk8y8ktqUnKHc5UwFMbkpcyrOTZMTtUcsLq4dfUGX06Mp
l7/VuY2w3O0N+7yXcHG1mc54dSWvF04e1Q6UAPqICdISqyp/omKHWCwMJzNU
dHDDcYVO2RkLK498DgHTDPHLB9cmbkQ0bFbW5m9X2RhW/mzDxZ41qKQGPoyJ
BukvYOMdvtICv/yLDQRDPCN3WoduK8/JDw3MV86Nd0vGJyNzcWA+4cyM0gwI
AnnqYKBOohN7gmkhErlN3+iCNvGr61X8exgywfaURR9H6ZP3zWr3/0WoDf8B
BN6QfQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[228]=",
 CellID->325192106]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Convergence rate", "Subsection",
 CellTags->"DoubleExponentialConvergenceRate",
 CellID->175923852],

Cell[TextData[{
 "In this section is demonstrated that the asymptotic error of the ",
 "double-exponential",
 " quadrature in terms of the number ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " of evaluation points used is "
}], "Text",
 CellID->114304679],

Cell[BoxData[
 FormBox[
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"c", " ", "n"}], 
      RowBox[{"log", "(", "n", ")"}]]}]], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"ConvergenceRate1",
 CellID->183112326],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 " is a positive constant."
}], "Text",
 CellID->178350123],

Cell["\<\
This defines a double-exponential integration function that returns integral \
estimate and number of points used.\
\>", "MathCaption",
 CellID->264295893],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"DERuleEstimate", "[", 
    RowBox[{"f_", ",", 
     RowBox[{"{", 
      RowBox[{"a_", ",", "b_"}], "}"}], ",", "h_", ",", 
     RowBox[{"wprec_:", "MachinePrecision"}]}], "]"}], ":=", 
   "\[IndentingNewLine]", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"$MaxExtraPrecision", "=", "50000"}], ",", "\[Phi]", ",", "F", 
       ",", "i", ",", "j", ",", "temp", ",", "s1", ",", "s2"}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{
       RowBox[{"\[Phi]", "[", "t_", "]"}], ":=", 
       RowBox[{"Rescale", "[", 
        RowBox[{
         RowBox[{
          FractionBox["1", "2"], 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"Tanh", "[", 
             RowBox[{
              FractionBox["1", "2"], "\[Pi]", " ", 
              RowBox[{"Sinh", "[", "t", "]"}]}], "]"}], "+", "1"}], ")"}]}], 
         ",", 
         RowBox[{"{", 
          RowBox[{"0", ",", "1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"a", ",", "b"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"F", "[", "t_", "]"}], ":=", 
       RowBox[{"Evaluate", "[", 
        RowBox[{
         RowBox[{"f", "[", 
          RowBox[{"\[Phi]", "[", "t", "]"}], "]"}], "*", 
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"\[Phi]", "[", "t", "]"}], ",", "t"}], "]"}]}], "]"}]}], 
      ";", "\[IndentingNewLine]", 
      RowBox[{"i", "=", "1"}], ";", "\[IndentingNewLine]", 
      RowBox[{"s1", "=", 
       RowBox[{"FixedPoint", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"temp", "=", 
             RowBox[{"F", "[", 
              RowBox[{"i", "*", "h"}], "]"}]}], ";", 
            RowBox[{"i", "++"}], ";", 
            RowBox[{"N", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"N", "[", 
                RowBox[{"temp", ",", 
                 RowBox[{"3", "*", "wprec"}]}], "]"}], "+", "#1"}], ",", 
              "wprec"}], "]"}]}], ")"}], "&"}], ",", "0"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"j", "=", 
       RowBox[{"-", "1"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"s2", "=", 
       RowBox[{"FixedPoint", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"temp", "=", 
             RowBox[{"F", "[", 
              RowBox[{"j", "*", "h"}], "]"}]}], ";", 
            RowBox[{"j", "--"}], ";", 
            RowBox[{"N", "[", 
             RowBox[{
              RowBox[{
               RowBox[{"N", "[", 
                RowBox[{"temp", ",", 
                 RowBox[{"3", "*", "wprec"}]}], "]"}], "+", "#1"}], ",", 
              "wprec"}], "]"}]}], ")"}], "&"}], ",", "0"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"i", "-", "j", "+", "1"}], ",", 
        RowBox[{"h", 
         RowBox[{"(", 
          RowBox[{"s1", "+", 
           RowBox[{"F", "[", "0", "]"}], "+", "s2"}], ")"}]}]}], "}"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[229]:=",
 CellID->4842064],

Cell["This defines a function.", "MathCaption",
 CellID->36256814],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  FormBox[
   RowBox[{
    FractionBox["1", 
     SuperscriptBox["x", 
      RowBox[{"1", "/", "4"}]]], " ", 
    RowBox[{"Log", "[", 
     FractionBox["1", "x"], "]"}]}],
   TraditionalForm]}]], "Input",
 CellLabel->"In[230]:=",
 CellID->118848118],

Cell["This is the exact integral.", "MathCaption",
 CellID->420157104],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[231]:=",
 CellID->123022023],

Cell[BoxData[
 FractionBox["16", "9"]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[231]=",
 CellID->338331732]
}, Open  ]],

Cell["\<\
This finds the errors and number of evaluation points for a range of step \
sizes of the trapezoidal rule.\
\>", "MathCaption",
 CellID->113371595],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b"}], "}"}], "=", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"wprec", "=", "30"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"range", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     FractionBox["1", "i"], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "2", ",", "7"}], "}"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"range", "=", 
   RowBox[{"Join", "[", 
    RowBox[{"range", ",", 
     RowBox[{"Mean", "/@", 
      RowBox[{"Partition", "[", 
       RowBox[{"range", ",", "2", ",", "1"}], "]"}]}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"range", "=", 
   RowBox[{"Sort", "[", 
    RowBox[{"range", ",", "Greater"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"err", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"DERuleEstimate", "[", 
       RowBox[{"f", ",", 
        RowBox[{"{", 
         RowBox[{"a", ",", "b"}], "}"}], ",", "#1", ",", "wprec"}], "]"}], 
      "&"}], ",", "range"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"err", "=", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"#1", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}],
          ",", 
         RowBox[{"Abs", "[", 
          RowBox[{"exact", "-", 
           RowBox[{
           "#1", "\[LeftDoubleBracket]", "2", "\[RightDoubleBracket]"}]}], 
          "]"}]}], "}"}], "&"}], ",", "err"}], "]"}]}], ";"}], " ", 
  RowBox[{"(*", " ", "errors", " ", "*)"}]}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"logErr", "=", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"#1", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}],
          ",", 
         RowBox[{"Log", "[", 
          RowBox[{
          "#1", "\[LeftDoubleBracket]", "2", "\[RightDoubleBracket]"}], 
          "]"}]}], "}"}], "&"}], ",", "err"}], "]"}]}], ";"}], " ", 
  RowBox[{"(*", " ", 
   RowBox[{"logarithm", " ", "of", " ", "the", " ", "errors"}], " ", 
   "*)"}]}], "\n", 
 RowBox[{
  RowBox[{"points", "=", 
   RowBox[{"First", "/@", "err"}]}], ";"}]}], "Input",
 CellLabel->"In[232]:=",
 CellID->260432712],

Cell[TextData[{
 "This fits ",
 Cell[BoxData[
  FormBox[
   FractionBox["x", 
    RowBox[{"Log", "[", "x", "]"}]], TraditionalForm]], "InlineMath"],
 " through the logarithms of the errors; see (",
 
 CounterBox["NumberedEquation", "ConvergenceRate1"],
 ")."
}], "MathCaption",
 CellID->41900423],

Cell[BoxData[
 RowBox[{
  RowBox[{"p", "[", "x_", "]"}], ":=", 
  RowBox[{"Evaluate", "[", 
   RowBox[{"Fit", "[", 
    RowBox[{"logErr", ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", 
       FractionBox["x", 
        RowBox[{"Log", "[", "x", "]"}]]}], "}"}], ",", "x"}], "]"}], 
   "]"}]}]], "Input",
 CellLabel->"In[239]:=",
 CellID->459664202],

Cell["\<\
Here is the fitted function.The summation term 30.48 is just a translation \
parameter.\
\>", "MathCaption",
 CellID->885589942],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"p", "[", "x", "]"}]], "Input",
 CellLabel->"In[240]:=",
 CellID->59134763],

Cell[BoxData[
 RowBox[{"30.4819289811061055058`4.277348568260877", "-", 
  FractionBox[
   RowBox[{"6.4970711652401756355`4.601488086948186", " ", "x"}], 
   RowBox[{"Log", "[", "x", "]"}]]}]], "Output",
 ImageSize->{101, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[240]=",
 CellID->622520259]
}, Open  ]],

Cell[TextData[{
 "You see that the errors fit the model (",
 
 CounterBox["NumberedEquation", "ConvergenceRate1"],
 "):"
}], "MathCaption",
 CellID->121026898],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"logErr", ",", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"#1", ",", 
         RowBox[{"p", "[", "#1", "]"}]}], "}"}], "&"}], "/@", "points"}]}], 
    "}"}], ",", 
   RowBox[{"PlotRange", "->", "All"}], ",", 
   RowBox[{"PlotStyle", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "Red", "}"}], ",", 
      RowBox[{"{", "Blue", "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[241]:=",
 CellID->47373384],

Cell[BoxData[
 GraphicsBox[{{}, {}, 
   {RGBColor[1, 0, 0], 
    LineBox[{{19., -12.288591860300176`}, {22., -15.76334109640622}, {
     26., -21.0242392777272}, {29., -25.167081864909974`}, {
     33., -30.322360433438188`}, {36., -34.79820615873645}, {
     40., -40.0008348201782}, {44., -44.26889811753689}, {
     47., -49.497152577969146`}, {51., -53.88887700193638}, {
     55., -58.98725904559414}}]}, 
   {RGBColor[0, 0, 1], 
    LineBox[{{19., -11.442645789504343`}, {22., -15.759935306454578`}, {
     26., -21.365475883952186`}, {29., -25.472484680470497`}, {
     33., -30.837357387890698`}, {36., -34.78759127693815}, {
     40., -39.968420465737104`}, {44., -45.061624309210465`}, {
     47., -48.8299250970753}, {51., -53.79205984216202}, {
     55., -58.68932221533763}}]}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{20., -50.},
  ImageMargins->0.,
  ImageSize->{257.333333333333, Automatic},
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{261, 164},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[241]=",
 CellID->294078889]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example implementation of double-exponential quadrature", "Subsection",
 CellTags->"DEExampleImplementation",
 CellID->5698822],

Cell[TextData[{
 "Following is an example implementation of the double-exponential quadrature \
with the finite region variable transformation. (Transformation (",
 
 CounterBox["NumberedEquation", "Double-exponentialStrategy3"],
 ") earlier.)"
}], "Text",
 CellID->14938677],

Cell[TextData[{
 "This is a definition of a function that applies the trapezoidal rule to a \
transformed integrand. The function uses (",
 
 CounterBox["NumberedEquation", "Double-exponentialStrategy4"],
 ") and it is made to reuse integral estimates computed with a twice larger \
step."
}], "MathCaption",
 CellID->18112252],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"IRuleEstimate", "[", 
    RowBox[{"F_", ",", "h_", ",", 
     RowBox[{"oldSum_:", "None"}]}], "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"$MaxExtraPrecision", "=", "50000"}], ",", "step", ",", "i", 
       ",", "temp", ",", "s1", ",", "s2"}], "}"}], ",", "\[IndentingNewLine]",
      
     RowBox[{
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"oldSum", "===", "None"}], ",", 
        RowBox[{"step", "=", "1"}], ",", 
        RowBox[{"step", "=", "2"}]}], "]"}], ";", "\[IndentingNewLine]", 
      RowBox[{"i", "=", "1"}], ";", "\[IndentingNewLine]", 
      RowBox[{"s1", "=", 
       RowBox[{"FixedPoint", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"temp", "=", 
             RowBox[{"F", "[", 
              RowBox[{"i", "*", "h"}], "]"}]}], ";", 
            RowBox[{"i", "+=", "step"}], ";", 
            RowBox[{"N", "[", 
             RowBox[{
              RowBox[{"N", "[", 
               RowBox[{"temp", ",", "60"}], "]"}], "+", "#1"}], "]"}]}], 
           ")"}], "&"}], ",", "0"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"i", "=", 
       RowBox[{"-", "1"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"s2", "=", 
       RowBox[{"FixedPoint", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"temp", "=", 
             RowBox[{"F", "[", 
              RowBox[{"i", "*", "h"}], "]"}]}], ";", 
            RowBox[{"i", "-=", "step"}], ";", 
            RowBox[{"N", "[", 
             RowBox[{
              RowBox[{"N", "[", 
               RowBox[{"temp", ",", "60"}], "]"}], "+", "#1"}], "]"}]}], 
           ")"}], "&"}], ",", "0"}], "]"}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"If", "[", 
       RowBox[{
        RowBox[{"oldSum", "===", "None"}], ",", 
        RowBox[{"h", 
         RowBox[{"(", 
          RowBox[{"s1", "+", 
           RowBox[{"F", "[", "0", "]"}], "+", "s2"}], ")"}]}], ",", 
        RowBox[{
         RowBox[{"h", 
          RowBox[{"(", 
           RowBox[{"s1", "+", "s2"}], ")"}]}], "+", 
         FractionBox["oldSum", "2"]}]}], "]"}]}]}], "\[IndentingNewLine]", 
    "]"}]}], ";"}]], "Input",
 CellLabel->"In[173]:=",
 CellID->213428876],

Cell[TextData[{
 "This is a definition of a simple double-exponential strategy, which finds \
the integral of the function ",
 Cell[BoxData[
  StyleBox["f", "TI"]], "InlineFormula"],
 " over the finite interval ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["a", "TI"], ",", 
    StyleBox["b", "TI"]}], "}"}]], "InlineFormula"],
 " with relative error ",
 Cell[BoxData[
  StyleBox["tol", "TI"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->315807871],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"Options", "[", "IStrategyDoubleExp", "]"}], "=", 
   RowBox[{"{", 
    RowBox[{"\"\<MaxRecursion\>\"", "\[Rule]", "7"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"IStrategyDoubleExp", "[", 
    RowBox[{"f_", ",", 
     RowBox[{"{", 
      RowBox[{"a_", ",", "b_"}], "}"}], ",", "tol_", ",", "opts___"}], "]"}], 
   ":=", "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Phi]", ",", "F", ",", "h", ",", "t", ",", "temp", ",", 
       RowBox[{"k", "=", "0"}], ",", "maxrec"}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"maxrec", "=", 
       RowBox[{
        RowBox[{"\"\<MaxRecursion\>\"", "/.", 
         RowBox[{"{", "opts", "}"}]}], "/.", 
        RowBox[{"Options", "[", "IStrategyDoubleExp", "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"\[Phi]", "[", "t_", "]"}], ":=", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"Rescale", "[", 
         RowBox[{
          RowBox[{
           FractionBox["1", "2"], 
           RowBox[{"(", 
            RowBox[{
             RowBox[{"Tanh", "[", 
              RowBox[{
               FractionBox["1", "2"], "\[Pi]", " ", 
               RowBox[{"Sinh", "[", "t", "]"}]}], "]"}], "+", "1"}], ")"}]}], 
          ",", 
          RowBox[{"{", 
           RowBox[{"0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"a", ",", "b"}], "}"}]}], "]"}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{
       RowBox[{"F", "[", "t_", "]"}], ":=", 
       RowBox[{"Evaluate", "[", 
        RowBox[{
         RowBox[{"f", "[", 
          RowBox[{"\[Phi]", "[", "t", "]"}], "]"}], "*", 
         RowBox[{"D", "[", 
          RowBox[{
           RowBox[{"\[Phi]", "[", "t", "]"}], ",", "t"}], "]"}]}], "]"}]}], 
      ";", "\[IndentingNewLine]", 
      RowBox[{"h", "=", "1"}], ";", "\[IndentingNewLine]", 
      RowBox[{"NestWhile", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{"temp", "=", 
             RowBox[{"IRuleEstimate", "[", 
              RowBox[{"F", ",", 
               RowBox[{"h", "/=", "2"}], ",", "#1"}], "]"}]}], ")"}], "&&", 
           RowBox[{
            RowBox[{"k", "++"}], "<", "maxrec"}]}], ")"}], "&"}], ",", 
        RowBox[{"IRuleEstimate", "[", 
         RowBox[{"F", ",", "h", ",", "None"}], "]"}], ",", 
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{
            RowBox[{"Abs", "[", "#1", "]"}], "*", "tol"}], "<=", 
           RowBox[{"Abs", "[", 
            RowBox[{"#1", "-", "#2"}], "]"}]}], ")"}], "&"}], ",", "2"}], 
       "]"}], ";", "\[IndentingNewLine]", "temp"}]}], "\[IndentingNewLine]", 
    "]"}]}], ";"}]}], "Input",
 CellLabel->"In[189]:=",
 CellID->582253814],

Cell[TextData[{
 "This defines a function that is singular at ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->71184663],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  FractionBox["1", 
   RadicalBox["x", "4"]]}]], "Input",
 CellLabel->"In[194]:=",
 CellID->59018210],

Cell["\<\
Here is the integral estimate from the double-exponential strategy.\
\>", "MathCaption",
 CellID->983676925],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"IStrategyDoubleExp", "[", 
   RowBox[{"f", ",", 
    RowBox[{"{", 
     RowBox[{"0", ",", "1"}], "}"}], ",", 
    SuperscriptBox["10", 
     RowBox[{"-", "8"}]]}], "]"}], "//", "InputForm"}]], "Input",
 CellLabel->"In[195]:=",
 CellID->10944481],

Cell["1.3333333333333333", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[195]//InputForm=",
 CellID->339348125]
}, Open  ]],

Cell["Here is the exact result.", "MathCaption",
 CellID->772004518],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{"f", "[", "x", "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}], "//", "N"}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[177]:=",
 CellID->813536014],

Cell["1.3333333333333333", "Output",
 GeneratedCell->False,
 CellAutoOverwrite->False,
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[177]//InputForm=",
 CellID->5871616]
}, Open  ]],

Cell["The two results are the same.", "Text",
 CellID->125133412],

Cell["This defines an oscillating function. ", "MathCaption",
 CellID->430093024],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  RowBox[{"Cos", "[", 
   RowBox[{"64", "*", 
    RowBox[{"Sin", "[", "x", "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[178]:=",
 CellID->155898858],

Cell["\<\
Here is the integral estimate given by the double-exponential strategy.\
\>", "MathCaption",
 CellID->3711478],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"res", " ", "=", 
   RowBox[{"IStrategyDoubleExp", "[", 
    RowBox[{"f", ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "\[Pi]"}], "}"}], ",", 
     SuperscriptBox["10", 
      RowBox[{"-", "8"}]]}], "]"}]}], ";", 
  RowBox[{"res", "//", "InputForm"}]}]], "Input",
 CellLabel->"In[179]:=",
 CellID->736800217],

Cell["0.29088010217372606", "Output",
 ImageSize->{133, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[179]//InputForm=",
 CellID->200115810]
}, Open  ]],

Cell["Here is the exact result.", "MathCaption",
 CellID->535222871],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[180]:=",
 CellID->141184157],

Cell[BoxData[
 RowBox[{"\[Pi]", " ", 
  RowBox[{"BesselJ", "[", 
   RowBox[{"0", ",", "64"}], "]"}]}]], "Output",
 ImageSize->{111, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[180]=",
 CellID->31452361]
}, Open  ]],

Cell["Here is the exact result in machine precision.", "MathCaption",
 CellID->529740859],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"exact", "//", "N"}], "//", "InputForm"}]], "Input",
 CellLabel->"In[181]:=",
 CellID->335479060],

Cell["0.2908801021737257", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[181]//InputForm=",
 CellID->134790997]
}, Open  ]],

Cell["The relative error is within the prescribed tolerance.", "MathCaption",
 CellID->690046052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"res", "-", "exact"}], "]"}], "/", "exact"}]], "Input",
 CellLabel->"In[182]:=",
 CellID->334013182],

Cell[BoxData["1.3358701943343305`*^-15"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[182]=",
 CellID->278863702]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"Trapezoidal\" Strategy", "Section",
 CellTags->"Trapezoidal",
 CellID->403370786],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 " strategy gives optimal convergence for analytic periodic integrands when \
the integration interval is exactly one period."
}], "Text",
 CellID->62448984],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<ExtraPrecision\>\"", "50", Cell[
    "maximum extra precision to be used internally", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->34816951],

Cell[TextData[{
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->236158595],

Cell[TextData[{
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 " takes the same options as ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 ". If the integration ranges are infinite or semi-infinite, ",
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 " becomes ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 "."
}], "Text",
 CellID->813545056],

Cell[TextData[{
 "For theoretical background, examples, and explanations of periodic \
functions integration (with trapezoidal quadrature) see [",
 ButtonBox["Weideman2002",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#253113814"],
 "]."
}], "Text",
 CellID->29033631],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["x"]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<Trapezoidal\>\"", ",", 
      RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[109]:=",
 CellID->768958397],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvi\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) iterated refinements in \\!\\(x\\) in the region \\!\\({\\({0.`, \
1.`}\\)}\\). NIntegrate obtained \\!\\(1.9771819583163235`\\) and \
\\!\\(0.009451548754043415`\\) for the integral and error estimates.\"\>"}]], \
"Message", "MSG",
 CellLabel->"During evaluation of (V6.0) In[109]:=",
 CellID->481205698],

Cell[BoxData["1.9771819583163235`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[109]=",
 CellID->336415460]
}, Open  ]],

Cell[TextData[{
 "Here is a table that shows the number of sampling points for different \
values of the parameter ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " used by ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 " respectively for the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "\[Pi]"], 
     RowBox[{
      FractionBox[
       RowBox[{"cos", "(", 
        RowBox[{
         RowBox[{"t", " ", 
          RowBox[{"sin", "(", "x", ")"}]}], "-", 
         RowBox[{"k", " ", "x"}]}], ")"}], "\[Pi]"], 
      RowBox[{"\[DifferentialD]", "x"}]}]}], ",", 
    RowBox[{"k", "=", "1"}]}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->153892033],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"k", "=", "1"}], ";"}], "\n", 
 RowBox[{
  RowBox[{"tab", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"t", ",", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
           "\"\<IntegralEstimate\>\"", ",", "\"\<Evaluations\>\"", ",", 
            "\"\<Timing\>\""}], "}"}], "/.", 
          RowBox[{"NIntegrateProfile", "[", 
           RowBox[{
            RowBox[{"NIntegrate", "[", 
             RowBox[{
              RowBox[{
               FractionBox["1", "\[Pi]"], 
               RowBox[{"Cos", "[", 
                RowBox[{
                 RowBox[{"t", " ", 
                  RowBox[{"Sin", "[", "x", "]"}]}], "-", 
                 RowBox[{"k", " ", "x"}]}], "]"}]}], ",", 
              RowBox[{"{", 
               RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}], ",", 
              RowBox[{"Method", "\[Rule]", "\"\<GaussKronrodRule\>\""}]}], 
             "]"}], ",", "1"}], "]"}]}], ")"}], "[", 
        RowBox[{"[", "2", "]"}], "]"}], ",", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
           "\"\<IntegralEstimate\>\"", ",", "\"\<Evaluations\>\"", ",", 
            "\"\<Timing\>\""}], "}"}], "/.", 
          RowBox[{"NIntegrateProfile", "[", 
           RowBox[{
            RowBox[{"NIntegrate", "[", 
             RowBox[{
              RowBox[{
               FractionBox["1", "\[Pi]"], 
               RowBox[{"Cos", "[", 
                RowBox[{
                 RowBox[{"t", " ", 
                  RowBox[{"Sin", "[", "x", "]"}]}], "-", 
                 RowBox[{"k", " ", "x"}]}], "]"}]}], ",", 
              RowBox[{"{", 
               RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}], ",", 
              RowBox[{"Method", "->", "\"\<Trapezoidal\>\""}]}], "]"}], ",", 
            "1"}], "]"}]}], ")"}], "[", 
        RowBox[{"[", "2", "]"}], "]"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"t", ",", "8", ",", "80", ",", "4"}], "}"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"TableForm", "[", 
  RowBox[{"tab", ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"None", ",", 
      RowBox[{"{", 
       RowBox[{
       "t", ",", "\"\<GlobalAdaptive\>\"", ",", "\"\<Trapezoidal\>\""}], 
       "}"}]}], "}"}]}]}], "]"}]}], "Input",
 CellLabel->"In[33]:=",
 CellID->102876005],

Cell[BoxData[
 TagBox[
  TagBox[GridBox[{
     {
      TagBox["t",
       HoldForm], 
      TagBox["\<\"GlobalAdaptive\"\>",
       HoldForm], 
      TagBox["\<\"Trapezoidal\"\>",
       HoldForm]},
     {"8", "143", "33"},
     {"12", "209", "33"},
     {"16", "275", "65"},
     {"20", "399", "65"},
     {"24", "457", "65"},
     {"28", "591", "65"},
     {"32", "743", "65"},
     {"36", "743", "65"},
     {"40", "741", "65"},
     {"44", "809", "129"},
     {"48", "1007", "129"},
     {"52", "941", "129"},
     {"56", "963", "129"},
     {"60", "1095", "129"},
     {"64", "1121", "129"},
     {"68", "1095", "129"},
     {"72", "1137", "129"},
     {"76", "1338", "129"},
     {"80", "1227", "129"}
    },
    GridBoxAlignment->{
     "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
      "RowsIndexed" -> {}},
    GridBoxDividers->{
     "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
      "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
    GridBoxSpacings->{"Columns" -> {
        Offset[0.27999999999999997`], {
         Offset[0.7]}, 
        Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
        Offset[0.2], {
         Offset[0.4]}, 
        Offset[0.2]}, "RowsIndexed" -> {}}],
   {None, OutputFormsDump`HeadedColumns}],
  Function[BoxForm`e$, 
   TableForm[
   BoxForm`e$, 
    TableHeadings -> {
     None, {$CellContext`t, "GlobalAdaptive", "Trapezoidal"}}]]]], "Output",
 ImageSize->{209, 299},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]//TableForm=",
 CellID->500815691]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example implementation", "Subsubsection",
 CellID->159230017],

Cell["\<\
This function makes a trapezoidal quadrature integral estimate with specified \
points.\
\>", "MathCaption",
 CellID->164913865],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"TrapStep", "[", 
    RowBox[{"f_", ",", 
     RowBox[{"{", 
      RowBox[{"a_", ",", "b_"}], "}"}], ",", 
     RowBox[{"n_", "?", "IntegerQ"}]}], "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"h", ",", "absc", ",", " ", "is"}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"h", "=", 
       FractionBox[
        RowBox[{"b", "-", "a"}], 
        RowBox[{"n", "-", "1"}]]}], ";", "\[IndentingNewLine]", 
      RowBox[{"absc", "=", 
       RowBox[{"Table", "[", 
        RowBox[{"i", ",", 
         RowBox[{"{", 
          RowBox[{"i", ",", "a", ",", "b", ",", "h"}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"is", "=", 
       RowBox[{"h", "*", 
        RowBox[{"Total", "[", 
         RowBox[{"MapAt", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"#", "/", "2"}], "&"}], ",", 
           RowBox[{"f", "/@", "absc"}], ",", 
           RowBox[{"{", 
            RowBox[{
             RowBox[{"{", "1", "}"}], ",", 
             RowBox[{"{", 
              RowBox[{"-", "1"}], "}"}]}], "}"}]}], "]"}], "]"}]}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"is", ",", "\[Infinity]", ",", "n"}], "}"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[242]:=",
 CellID->777780494],

Cell["\<\
This function improves a trapezoidal quadrature integral estimate using \
sampling points between the old ones.\
\>", "MathCaption",
 CellID->144281038],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"TrapStep", "[", 
    RowBox[{"f_", ",", 
     RowBox[{"{", 
      RowBox[{"a_", ",", "b_"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"oldEstimate_", ",", "oldError_", ",", "oldn_"}], "}"}]}], 
    "]"}], ":=", "\[IndentingNewLine]", 
   RowBox[{"Module", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"n", ",", "h", ",", "absc", ",", "is"}], "}"}], ",", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"n", "=", 
       RowBox[{
        RowBox[{"2", " ", "oldn"}], "-", "1"}]}], ";", "\[IndentingNewLine]", 
      
      RowBox[{"h", "=", 
       FractionBox[
        RowBox[{"b", "-", "a"}], 
        RowBox[{"n", "-", "1"}]]}], ";", "\[IndentingNewLine]", 
      RowBox[{"absc", "=", 
       RowBox[{"Table", "[", 
        RowBox[{"i", ",", 
         RowBox[{"{", 
          RowBox[{"i", ",", 
           RowBox[{"a", "+", "h"}], ",", 
           RowBox[{"b", "-", "h"}], ",", 
           RowBox[{"2", " ", "h"}]}], "}"}]}], "]"}]}], ";", 
      "\[IndentingNewLine]", 
      RowBox[{"is", "=", 
       RowBox[{
        RowBox[{"h", "*", 
         RowBox[{"Total", "[", 
          RowBox[{"f", "/@", "absc"}], "]"}]}], "+", 
        RowBox[{"oldEstimate", "/", "2"}]}]}], ";", "\[IndentingNewLine]", 
      RowBox[{"{", 
       RowBox[{"is", ",", 
        RowBox[{"Abs", "[", 
         RowBox[{"is", "-", "oldEstimate"}], "]"}], ",", "n"}], "}"}]}]}], 
    "\[IndentingNewLine]", "]"}]}], ";"}]], "Input",
 CellLabel->"In[257]:=",
 CellID->108065044],

Cell["This function is an interface to the precedign one.", "MathCaption",
 CellID->31061338],

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"Options", "[", "TrapezoidalIntegration", "]"}], "=", 
   RowBox[{"{", 
    RowBox[{"\"\<MaxRecursion\>\"", "\[Rule]", "7"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"TrapezoidalIntegration", "[", 
   RowBox[{"f_", ",", 
    RowBox[{"{", 
     RowBox[{"a_", ",", "b_"}], "}"}], ",", "tol_", ",", "opts___"}], "]"}], ":=",
   "\[IndentingNewLine]", 
  RowBox[{"Block", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"maxrec", ",", 
      RowBox[{"k", "=", "0"}], ",", "temp"}], "}"}], ",", 
    "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"maxrec", "=", 
      RowBox[{
       RowBox[{"\"\<MaxRecursion\>\"", "/.", 
        RowBox[{"{", "opts", "}"}]}], "/.", 
       RowBox[{"Options", "[", "TrapezoidalIntegration", "]"}]}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{
      RowBox[{"NestWhile", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{
           RowBox[{"(", 
            RowBox[{"temp", "=", 
             RowBox[{"TrapStep", "[", 
              RowBox[{"f", ",", 
               RowBox[{"{", 
                RowBox[{"a", ",", "b"}], "}"}], ",", "#"}], "]"}]}], ")"}], "&&",
            
           RowBox[{
            RowBox[{"k", "++"}], "<", "maxrec"}]}], ")"}], "&"}], ",", 
        RowBox[{"TrapStep", "[", 
         RowBox[{"f", ",", 
          RowBox[{"{", 
           RowBox[{"a", ",", "b"}], "}"}], ",", "5"}], "]"}], ",", 
        RowBox[{
         RowBox[{
          RowBox[{"#", "[", 
           RowBox[{"[", "2", "]"}], "]"}], ">", "tol"}], "&"}]}], "]"}], "[", 
      
      RowBox[{"[", "1", "]"}], "]"}], ";", "\[IndentingNewLine]", 
     RowBox[{"temp", "[", 
      RowBox[{"[", "1", "]"}], "]"}]}]}], "\[IndentingNewLine]", 
   "]"}]}]}], "Input",
 CellLabel->"In[272]:=",
 CellID->79780492],

Cell["Here is a definition of a (Bessel) function.", "MathCaption",
 CellID->85775916],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  RowBox[{
   FractionBox["1", "\[Pi]"], 
   RowBox[{"Cos", "[", 
    RowBox[{
     RowBox[{"80", " ", 
      RowBox[{"Sin", "[", "x", "]"}]}], "-", " ", "x"}], "]"}]}]}]], "Input",
 CellLabel->"In[269]:=",
 CellID->233005292],

Cell["Here is the trapezoidal quadrature estimate.", "MathCaption",
 CellID->118049408],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"res", "=", 
  RowBox[{
   RowBox[{"TrapezoidalIntegration", "[", 
    RowBox[{"f", ",", 
     RowBox[{"{", 
      RowBox[{"0", ",", "\[Pi]"}], "}"}], ",", 
     SuperscriptBox["10", 
      RowBox[{"-", "5"}]]}], "]"}], "//", "N"}]}]], "Input",
 CellLabel->"In[274]:=",
 CellID->15909342],

Cell[BoxData[
 RowBox[{"-", "0.05605729667571248`"}]], "Output",
 ImageSize->{75, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[274]=",
 CellID->292362605]
}, Open  ]],

Cell["Here is the exact value.", "MathCaption",
 CellID->37376087],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Pi]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[278]:=",
 CellID->80421202],

Cell[BoxData[
 RowBox[{"BesselJ", "[", 
  RowBox[{"1", ",", "80"}], "]"}]], "Output",
 ImageSize->{100, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[278]=",
 CellID->659098301]
}, Open  ]],

Cell["The relative error is within the prescribed tolerance.", "MathCaption",
 CellID->81918849],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Abs", "[", 
   RowBox[{"res", "-", "exact"}], "]"}], "/", "exact"}]], "Input",
 CellLabel->"In[279]:=",
 CellID->265174061],

Cell[BoxData[
 RowBox[{"-", "0.5727315324721072`"}]], "Output",
 ImageSize->{68, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[279]=",
 CellID->464743071]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Oscillatory Strategies", "Section",
 CellTags->"OscillatoryStrategy",
 CellID->775797877],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s oscillatory strategies are for one-dimensional integrals. Generally in \
quadrature, the algorithms for finite region integrals are different from the \
algorithms for infinite regions. ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " uses Chebyshev expansions of the integrand and the global adaptive \
integration strategy for finite region oscillatory integrals. For infinite \
oscillatory integrals ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " uses either a modification of the double-exponential algorithm or sequence \
summation acceleration over the sequence of integrals with regions between \
the integrand's zeros."
}], "Text",
 CellID->146769327],

Cell["Here is an example that uses both algorithms.", "Text",
 CellID->414054894],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"\[Piecewise]", GridBox[{
      {
       FractionBox[
        RowBox[{"Cos", "[", 
         RowBox[{"2000", " ", "x"}], "]"}], 
        SqrtBox["x"]], 
       RowBox[{"0", "<", "x", "<", "2"}]},
      {
       FractionBox[
        RowBox[{"Sin", "[", 
         RowBox[{"20", " ", "x"}], "]"}], 
        SqrtBox[
         SuperscriptBox["x", "2"]]], 
       RowBox[{"x", "<", "0"}]},
      {
       RowBox[{
        RowBox[{"BesselY", "[", 
         RowBox[{"2", ",", 
          SuperscriptBox["x", "3"]}], "]"}], "/", "x"}], 
       RowBox[{"x", ">", "2"}]}
     },
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.28], {
          Offset[0.84]}, 
         Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->248452077],

Cell[BoxData[
 RowBox[{"-", "1.5495988665487932`"}]], "Output",
 ImageSize->{54, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->87456733]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " automatically detects oscillatory (",
 "one-dimensional",
 ") integrands, and automatically decides which algorithm to use according to \
integrand's range."
}], "Text",
 CellID->106440972],

Cell["\<\
The integrals detected as being of oscillatory type have the form\
\>", "Text",
 CellID->41862045],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", "b"], 
    RowBox[{
     RowBox[{
      RowBox[{"k", "(", "x", ")"}], " ", 
      RowBox[{"f", "(", "x", ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->158552789],

Cell[TextData[{
 "in which the oscillating kernel ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " is of the form:"
}], "Text",
 CellID->277707479],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"cos", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"\[ImaginaryI]", " ", "\[Omega]", " ", 
     SuperscriptBox["x", "p"]}]], TraditionalForm]], "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"a", ",", "b"}], ")"}], TraditionalForm]], "InlineMath"],
 " finite;"
}], "ItemizedText",
 CellID->249894444],

Cell[CellGroupData[{

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"cos", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"\[ImaginaryI]", " ", "\[Omega]", " ", 
     SuperscriptBox["x", "p"]}]], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["J", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     SubscriptBox["H", "\[Nu]"], 
     RowBox[{"(", "1", ")"}]], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     SubscriptBox["H", "\[Nu]"], 
     RowBox[{"(", "2", ")"}]], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["j", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["y", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"(", 
    RowBox[{"a", ",", "b"}], ")"}], TraditionalForm]], "InlineMath"],
 " infinite or semi-infinite."
}], "ItemizedText",
 CellID->4970220],

Cell[TextData[{
 "In these oscillating kernel forms ",
 Cell[BoxData[
  FormBox["\[Omega]", TraditionalForm]]],
 ", ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["\[Nu]", TraditionalForm]], "InlineMath"],
 " are real constants, and ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 " is a positive integer."
}], "Text",
 CellID->538601635]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Finite Region Oscillatory Integration", "Section",
 CellID->144042466],

Cell[TextData[{
 "Modified Clenshaw\[Dash]Curtis quadrature ([",
 ButtonBox["PiesBrand75",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#631520174"],
 "][",
 ButtonBox["PiesBrand84",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#838261"],
 "]) is for finite region ",
 "one-dimensional",
 " integrals of the form"
}], "Text",
 CellID->629911635],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", "b"], 
    RowBox[{"sin", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"\[Omega]", " ", 
        SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], " ", 
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", "b"], 
    RowBox[{
     RowBox[{"cos", "(", 
      RowBox[{
       RowBox[{"\[Omega]", " ", 
        SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], " ", 
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", " ", 
   RowBox[{"or", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "a", "b"], 
     RowBox[{
      RowBox[{"exp", "(", 
       RowBox[{
        RowBox[{"\[ImaginaryI]", " ", "\[Omega]", " ", 
         SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], " ", 
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"FiniteRegionOscillatoryIntegration1",
 CellID->17514027],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"a", ",", " ", "b", ",", " ", "k", ",", " ", "c", ",", " ", "p"}], 
   TraditionalForm]]],
 " are finite real numbers."
}], "Text",
 CellID->823338812],

Cell[TextData[{
 "The modified Clenshaw\[Dash]Curtis quadrature rule approximates ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " with a single polynomial through Chebyshev polynomials expansion. This \
leads to simplified computations because of the orthogonality of the \
Chebyshev polynomials with sin and cos functions. The modified \
Clenshaw\[Dash]Curtis quadrature rule is used with the strategy ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ". For smooth ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " the modified Clenshaw\[Dash]Curtis quadrature is usually superior [",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "] to other approaches for oscillatory integration (as Filon's quadrature \
and multi-panel integration between the zeros of the integrand)."
}], "Text",
 CellID->480652614],

Cell[TextData[{
 "Modified Clenshaw\[Dash]Curtis quadrature is quite good for highly \
oscillating integrals of the form (",
 
 CounterBox["NumberedEquation", "FiniteRegionOscillatoryIntegration1"],
 "). For example, modified ",
 "Clenshaw\[Dash]Curtis",
 " quadrature uses less than hundred integrand evaluations for both ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"sin", " ", 
     RowBox[{"(", 
      RowBox[{"200", " ", "x"}], ")"}]}], 
    SuperscriptBox["x", "2"]], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"sin", " ", 
     RowBox[{"(", 
      RowBox[{"20000", " ", "x"}], ")"}]}], 
    SuperscriptBox["x", "2"]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->785968836],

Cell["\<\
Number of integrand evaluations for modified Clenshaw\[Dash]Curtis quadrature \
for slowly oscillating kernel.\
\>", "MathCaption",
 CellID->316815895],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"200", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}], ",", 
    RowBox[{"EvaluationMonitor", ":>", 
     RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->399327677],

Cell[BoxData["78"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->63739472]
}, Open  ]],

Cell["\<\
Timing and integral estimates for modified Clenshaw\[Dash]Curtis quadrature \
for slowly oscillating kernel.\
\>", "MathCaption",
 CellID->167970950],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"200", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->13860375],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.16999999999999998`", ",", 
   RowBox[{"-", "0.07777386322872125`"}]}], "}"}]], "Output",
 ImageSize->{128, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->52374568]
}, Open  ]],

Cell["\<\
Number of integrand evaluations for modified Clenshaw\[Dash]Curtis quadrature \
for highly oscillating kernel.\
\>", "MathCaption",
 CellID->141395926],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"20000", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}], ",", 
    RowBox[{"EvaluationMonitor", ":>", 
     RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->308035032],

Cell[BoxData["78"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->417392447]
}, Open  ]],

Cell["\<\
Timing and integral estimates for modified Clenshaw\[Dash]Curtis quadrature \
for highly oscillating kernel.\
\>", "MathCaption",
 CellID->14385849],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"20000", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->563535983],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.11100000000000017`", ",", 
   RowBox[{"-", "0.0009168934843555823`"}]}], "}"}]], "Output",
 ImageSize->{149, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->553968583]
}, Open  ]],

Cell[TextData[{
 "On the other hand, without symbolic preprocessing, the default ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " method\[LongDash]",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " strategy with a Gauss\[Dash]Kronrod rule\[LongDash]uses thousands of \
evaluations for ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"sin", "(", 
     RowBox[{"200", "x"}], ")"}], 
    SuperscriptBox["x", "2"]], TraditionalForm]], "InlineMath"],
 ", and it cannot integrate ",
 Cell[BoxData[
  FormBox[
   FractionBox[
    RowBox[{"sin", " ", 
     RowBox[{"(", 
      RowBox[{"20000", " ", "x"}], ")"}]}], 
    SuperscriptBox["x", "2"]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->93496122],

Cell["\<\
Number of integrand evaluations for Gaussian quadrature for slowly \
oscillating kernel.\
\>", "MathCaption",
 CellID->9239284],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"200", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"Automatic", ",", " ", 
       RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}], ",",
     
    RowBox[{"EvaluationMonitor", ":>", 
     RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->105980387],

Cell[BoxData["2656"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->27338483]
}, Open  ]],

Cell["\<\
Timing and integral estimates for Gaussian quadrature for slowly oscillating \
kernel.\
\>", "MathCaption",
 CellID->434742908],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"200", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"Automatic", ",", " ", 
       RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
   "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->997007794],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.19999999999999982`", ",", 
   RowBox[{"-", "0.07777386319469248`"}]}], "}"}]], "Output",
 ImageSize->{121, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->401188821]
}, Open  ]],

Cell["\<\
Number of integrand evaluations for Gaussian quadrature for highly \
oscillating kernel.\
\>", "MathCaption",
 CellID->383064431],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"20000", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"Automatic", ",", " ", 
       RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}], ",",
     
    RowBox[{"EvaluationMonitor", ":>", 
     RowBox[{"k", "++"}]}]}], "]"}], ";", "k"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->122169534],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[9]:=",
 CellID->144131877],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({0.3301060999529583`}\\). NIntegrate obtained \
\\!\\(-0.09057436222462581`\\) and \\!\\(0.42924020409664687`\\) for the \
integral and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[9]:=",
 CellID->2482478],

Cell[BoxData["1290"], "Output",
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->11717390]
}, Open  ]],

Cell["\<\
Timing and integral estimates for Gaussian quadrature for highly oscillating \
kernel.\
\>", "MathCaption",
 CellID->23680421],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{"20000", " ", "x"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      FractionBox["2", "10"], ",", "2"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"Automatic", ",", " ", 
       RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
   "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->108059302],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[10]:=",
 CellID->886284367],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({0.3301060999529583`}\\). NIntegrate obtained \
\\!\\(-0.09057436222462581`\\) and \\!\\(0.42924020409664687`\\) for the \
integral and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[10]:=",
 CellID->311316561],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.3909999999999998`", ",", "0``0.3672996073034838"}], 
  "}"}]], "Output",
 ImageSize->{114, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->99434998]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Extrapolating Oscillatory Strategy ", "Section",
 CellID->50593814],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<ExtrapolatingOscillatory\>\""], "InlineFormula"],
 " strategy is for oscillating integrals in infinite ",
 "one-dimensional",
 " regions. The strategy uses sequence convergence acceleration for the sum \
of the sequence that consists of each of the integrals with regions between \
two consecutive zeros of the integrand."
}], "Text",
 CellID->307795057],

Cell[TextData[{
 "Here is an example of an integration using ",
 Cell[BoxData["\"\<ExtrapolatingOscillatory\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->170767220],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{
      RowBox[{"200", " ", 
       RowBox[{"x", "^", "2"}]}], "+", "5"}], "]"}], 
    FractionBox["1", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "+", "1"}], ")"}], "^", "2"}]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<ExtrapolatingOscillatory\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[294]:=",
 CellID->12526274],

Cell[BoxData[
 RowBox[{"-", "0.030972137666436736`"}]], "Output",
 ImageSize->{75, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[294]=",
 CellID->40138]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], "GlobalAdaptive", Cell[TextData[{
     "integration strategy used to integrate between the zeros and which will \
be used if ",
     StyleBox["ExtrapolatingOscillatory", "InlineCode"],
     " fails"
    }], "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic processing ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->376324318],

Cell["Consider the integral", "Text",
 CellID->37721581],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", "\[Infinity]"], 
    RowBox[{
     RowBox[{
      RowBox[{"k", "(", "x", ")"}], " ", 
      RowBox[{"f", "(", "x", ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"ExtrapolatingOscillatoryStrategy1",
 CellID->253417908],

Cell[TextData[{
 "where the function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " is the oscillating kernel and the function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"f", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " is smooth. Let ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["z", "i"], TraditionalForm]], "InlineMath"],
 " be the zeros of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " enumerated from the lower (finite) integration bound, that is, the \
inequality ",
 Cell[BoxData[
  FormBox[
   RowBox[{"a", "\[LessEqual]", " ", 
    SubscriptBox["z", "1"], "<", " ", 
    SubscriptBox["z", "2"], "<", "\[Ellipsis]", "<", " ", 
    SubscriptBox["z", "i"], " ", "<", "\[Ellipsis]"}], TraditionalForm]], 
  "InlineMath"],
 " holds. If the integral (",
 
 CounterBox["NumberedEquation", "ExtrapolatingOscillatoryStrategy1"],
 ") converges then the sequence"
}], "Text",
 CellID->391745360],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", 
     SubscriptBox["z", "1"]], 
    RowBox[{
     RowBox[{"k", "(", " ", "x", ")"}], " ", "f", " ", 
     RowBox[{"(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", 
     SubscriptBox["z", "2"]], 
    RowBox[{
     RowBox[{
      RowBox[{"k", "(", "x", ")"}], " ", 
      RowBox[{"f", "(", "x", ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", "\[Ellipsis]", ",", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", 
     SubscriptBox["z", "i"]], 
    RowBox[{
     RowBox[{"k", "(", "x", ")"}], " ", 
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", "\[Ellipsis]"}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"ExtrapolatingOscillatoryStrategy2",
 CellID->285034341],

Cell[TextData[{
 " converges too. The elements of the sequence (",
 
 CounterBox["NumberedEquation", "ExtrapolatingOscillatoryStrategy2"],
 ") are the partial sums of the sequence"
}], "Text",
 CellID->30601033],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", "a", 
     SubscriptBox["z", "1"]], 
    RowBox[{
     RowBox[{"k", "(", "x", ")"}], " ", 
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     SubscriptBox["z", "1"], 
     SubscriptBox["z", "2"]], 
    RowBox[{
     RowBox[{
      RowBox[{"k", "(", "x", ")"}], " ", 
      RowBox[{"f", "(", "x", ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", "\[Ellipsis]", ",", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     SubscriptBox["z", 
      RowBox[{"i", "-", "1"}]], 
     SubscriptBox["z", "i"]], 
    RowBox[{
     RowBox[{"k", "(", "x", ")"}], " ", 
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], ",", 
   RowBox[{"\[Ellipsis]", "."}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"ExtrapolatingOscillatoryStrategy3",
 CellID->43379409],

Cell[TextData[{
 "Often a good estimate of the limit of the sequence (",
 
 CounterBox["NumberedEquation", "ExtrapolatingOscillatoryStrategy2"],
 ") can be computed with relatively few elements of it through some \
convergence acceleration technique."
}], "Text",
 CellID->115358430],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<Oscillatory\>\""], "InlineFormula"],
 " ",
 "strategy uses ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 " with Wynn's extrapolation method for the integrals in (",
 
 CounterBox["NumberedEquation", "ExtrapolatingOscillatoryStrategy3"],
 "). Each integral in (",
 
 CounterBox["NumberedEquation", "ExtrapolatingOscillatoryStrategy3"],
 ") is calculated by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " without oscillatory methods."
}], "Text",
 CellID->490306447],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<Oscillatory\>\""], "InlineFormula"],
 " strategy applies its algorithm to oscillating kernels ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " in (",
 
 CounterBox["NumberedEquation", "ExtrapolatingOscillatoryStrategy1"],
 ") that are of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"cos", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["J", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Y", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     SubscriptBox["H", "\[Nu]"], 
     RowBox[{"(", "1", ")"}]], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox[
     SubscriptBox["H", "\[Nu]"], 
     RowBox[{"(", "2", ")"}]], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["j", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["y", "\[Nu]"], "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Omega]", ","}], TraditionalForm]], "InlineMath"],
 " ",
 Cell[BoxData[
  FormBox["c", TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox["\[Nu]", TraditionalForm]], "InlineMath"],
 " are real constants."
}], "Text",
 CellID->502247788],

Cell[CellGroupData[{

Cell["Example implementation", "Subsubsection",
 CellID->64282444],

Cell[TextData[{
 "The following example implementation illustrates how ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<Oscillatory\>\""], "InlineFormula"],
 " ",
 "strategy works."
}], "Text",
 CellID->101549673],

Cell[TextData[{
 "Here is a definition of an oscillation function that will be integrated in \
the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "\[Infinity]"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ". The zeros of the oscillating function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", "(", 
    RowBox[{"\[Omega]", " ", "x"}], ")"}], TraditionalForm]], "InlineMath"],
 " are ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"i", 
     FractionBox["1", "\[Omega]"], " ", "\[Pi]"}], ",", " ", 
    RowBox[{"i", "\[Element]", "\[DoubleStruckCapitalN]"}]}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->264772130],

Cell[BoxData[{
 RowBox[{
  RowBox[{"Clear", "[", 
   RowBox[{"\[Omega]", ",", "k", ",", "f"}], "]"}], 
  ";"}], "\[IndentingNewLine]", 
 RowBox[{
  RowBox[{"\[Omega]", "=", "20"}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"k", "[", "x_", "]"}], ":=", 
   RowBox[{"Sin", "[", 
    RowBox[{"\[Omega]", " ", "x"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", "x_", "]"}], ":=", 
   FractionBox["1", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"x", "+", "1"}], ")"}], "^", "2"}]]}], ";"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->338304790],

Cell[TextData[{
 "Here is a plot of the oscillatory function in the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"0", ",", "10"}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->199048612],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"k", "[", "x", "]"}], 
    RowBox[{"f", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"PlotPoints", "->", "1000"}], ",", 
   RowBox[{"PlotRange", "->", "All"}]}], "]"}]], "Input",
 CellLabel->"In[89]:=",
 CellID->330511491],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNXEuoZkcRPvM/7j8qvkBQs9CAWSiooHERIZNRRx1Fx+BCZiEuJoMws9CR
OG50oxtnYza6MSBm4QPULBKEBCGCGBQDYkAcECNIhDAuDAGDixHh+HV/1VXV
dfqc8//3/nfGy50759FdXV1dj6/q9DnnLly99PkvXLh6+eKFO88+eOFLly5f
/PKdH73yIC4tT3TdiUtd111/U4fjHkf53xv7vu+W6Sj/5P/uTwddulNu2JXx
tsd3Z5XPX3755nN/+Wdp+bGJlo98//cXH3h0m5bXvvmr06ce3qbllS/+HC1v
3vzvli1v3PjXbMvzn/nxp+//wfU//WO25Sc+/ghm9OwfXphtiaG/9tVfPP3r
v4WWqaG2wLSffOLP29CabbnMzSGXj5z53ne+/duf/fSPc20hGUwca4TfubaQ
zeVLjz3+2PWHvvX0XFtIB6IHt+D5drWF3CH9Xz711298/am5tqRI6odru8Hf
BZouXR/87dK1gwk6XAGjk9uurC0pD3oPqHFdIjXe5E/ipFvzXIk3qa9LSyoG
iTq9OxyPC5DqlsphvnV2hN5apNmS3K50fP94Pk2vrGn5v9W7W6ml0XtN6g5N
eO9t19oWll/84tmJ9jAJGAYOMefiHafaQ9Wh8LegPTwrAous+4sv/hs+ufS/
b5v+PJD+OIKzLj7T+i+LYYz0w3JC5j/64bOz437usz95/vmXYn8oFPiGOc72
p3hif/T8ytUn4Ly27y+xIfVf4DhHPmmZ/ty7I43cHyfF7nfloVtiIeGAD9+f
V7EMs/2p8ofvL+bV9xA7vKh4EVLFMkIJ081T4xSWSqFLvoT9eU3ntkP/7GXW
jOJ9ZUG70lCfhrhdkM2haYiG1jTUkpp9YRowEBxCm5/53d8nxl+kPwe0d16C
DcKxko7GWllhRhtZGNHwuyfoWlTse0bdKbrQXJj+EeguFZ+YPiyxnhARaGcS
79sTVfhKSBiKsk+qK6wVVsx0/zCyVZzIi2JVvKpec5Iy8SYvkaOMCnK0runy
mvryTOa9E5QNu/U9j8col+iu8XWS4xm6BKRClxQNpe9O10V9XhTKTDkM0++P
MiXMNTzZi1YLvfTn7XsbiTCAC0GrxKgOn71DR0oDcQTLCvqeWhlGEBzLEehW
qVccAQew0IQ+6hGGcxkbSTWwGomRhNrMkQBKAMBSrNnvSPTBvMQryNMxGNzZ
w9995shj0T0TBa9JOXJAj09FP/J4fYa+XYhDcpXjUWnMU2w7HqdCyhaj58fz
ORvXEksL8RbM/dbx8ZfqEeXcIuZynI/KbiBwiN18imIEwTyQO87TrWleUu8Y
tfP5geeB/n814IJtKCWoFWMlWRM+4NVgTPjdCx88Jh+0V15nFAk2zouGYZHl
JJ09LB8WA2s+vC5RX8kHfW9f4/kVqfAvtQayknHSnzccgjM910kPOSPsClbF
1K/yGPS25AyGxHrJ1pzpWjU446jkzLxQAVzB/gSFBezoUf0KbIJZKtt++Uvt
aZeqS1J5Ux1wyFbtgQFcuCNmpdRr7jRKLdVi5HxL6a20n/C6DlphvKkG0OEI
b0xuiAq40tACyLJ4rleOStJLJPLMc40Fcq7VhVjJEN6pD+q9+ho5Uoo4ZOlO
+GcKw0vkH+oBAZUoehT+rfYT23NNPP/kjjPwMYKcVXiPYYn8r6lFGqlymSqr
ug+ZncYULHfyp+8J80nZU6pEfUD+v8/pllaotpQCJbq9FNS36XoNpdBEdg2M
Qo3J45zUWmlfIDKdJWUBpYeWJtlEWdxbxCjnlMbclUVLNltKzDKMtsRoYyYx
j2S8xHw0pUZQYmo3NA1BAF6+9HxebvhR10AdU3th+Qu0NoU7ZgTdXfiThHLq
tktQvbz67NVAcoz5Q8k5zFi1p4zG8EJemQ35VI2W+kRAtvLsxufEkKpGSowv
c7vDzb1ryux0T/s8fazSVOkEaTK+kxTb8Lo9GykWSWlafj6OeigyL00dRepo
jFC6Aq6+wHkQDyjqYMGxY14Jzphadq/ra083lODxauiYTD1CdTL19ZR8PoUo
XXzfDNFRV2Yi8lXUwxAk2qr9KOgquyVa4hX6GdaLKHOQEc96sqyUzPzAa6uT
2L69Q5S12mNT9pljIoNGlkBvsArt2M8Q3BBjcUxiLKvGSBQ/4JzIN6PSoshK
tDr3XKvn9znygXqSfpi79VrZ1Xov2fRPQFJ+i1Xz+By8gOWEDbhqaqjd3W7V
isWUJ5Jz2CCuRLQKzejl3LScEjVE7leihcHZXyu9I1hDI2FfR05fE/T+X7FM
X/t/q0TUeVrTVxmqr5AzPR0paZUDBNAIrRVLw65S1ZCrApOUWKv5nmCtR4+w
CtG3xFXweu3tIeYcR4zCS40bIz7OawNXZpgB0tY0EyBQ6QsCDNUFQZBca+gO
dQAdcBWnbK02KE87NJpjbmhaPd0BWTAA4eFAK2Xoh1/NjzBj6Gh6Csg1RW96
B3v+IFJ86PjW1Gd4zfPd4lfG01yDhg+lHqxCu3bOFC04Yt9dLdhjFB4PMQfH
UJ3B+FQznz2O6gxO0CfojHr8CgFSDmiEAS27p8LgF1cPXDuIF71THs12UCAs
EiYJ5ngFrKARrqCDxl70SRWiTbFAdEt1GeoarEAyaK9raQz1+oKPPrxH3Yte
3uqLbV2M9HaNGvrkti/4cxV0qT0Py6LbuMDrktfpiAvm6hiqHY0M2PnDjY8y
oAG9TCs1xIm+GubrOb6moJqKM+K3dT22avAww3FYJtkH8R7OaMukg7FYhaZj
QDMdlykjhIvrWuEFY2jPUcxCQBfnUFkMmeDMuozJfQswfpoEW0OVwTKoCrbh
VY6IxYG8rCIJXqEW+AXHtDKuNnQPVNHH/DpYQ28Ml7w9e2P6GJipxEEZBfzj
CthMqCn3Tnvk0ILMb45gQVGjore1CgEtSDX+kBbqsqmmhWrklftaixxU5ram
z8g8wv+uHiAiB/XiI+NFepH/yG+DPo1lhL5ZeDsDoQW21puZr/c0arm6q4xj
xDobOuBSateMyBuPuog0ld+Nj4a4ymdfMhfoGujaXIP32nAzJPtCbjit6EJu
XFqMCbrJTnmPdDkFrBTapZi4LPIEPykbXWRimAv6pvVkXwxIHvtcIYW5Kr8b
jAO6oIUDCB3HEAPvnQQFWC1YZFKKMUAGU1NuQQkhArcgZhwnz7R00k1IfAGi
tO3XZ4vuuvP4d6K29spaamu0FUwejsWd7axcEjvNhus4o3Q3hsszdUEspGEx
bqgVaM5cUdrGJ4vRStBe9gVGKyFtjbd9icOd7uuJFmAyYm0/cVycpmUOtku4
BJrc+xVs6vG+eQD2tNifE12NZ8pHX+Jls/bEZhKPdJ5MAjrZdZ3iFE65JP4Z
GRrQEkFdo9qa6A2MgK9UYfEz4Q94hsKxnmIZPWstaIe/FvUwNqaFpYXgUpFn
VVaIFtItsgqAovhZ1hcwAW5PSFskurLTFQxBdtyMrntfQZ3xL+t+ivcMfkJP
ZJDqQZANtwxrBpQMu+twi6iT20gSV57LzFBtRRWKamu7WVNEb2ZN3l+OWc8g
Zm0sRmT2RIGXZQYSf4ZPcmgVqyZP0WejrWCXtm/VuNBrIa5Y0EHQdoy5Lpqu
AvXI0O+CozWEer+at8+VzU4zq/LUiRZEfTfsxLbUa8VOazpyPjMvD6fyE1f0
5Z441TGwBV0s1duU32Ag6myz/iceKlm55jNUG8lnwIWs8apYIvujS2r1qkD1
NeH81eH8LUfW12HVJD7tiXobsVzMliI2MawxrN5QXsNMgd5E38GpspVY5fQZ
OL3c0JO6zMB2CMhOLF5VhCWVzQPXNqAr01AdgkZDvQE/GI6caF7M6iauaB0N
FFM0p17ArgS/qt7Ie0fxzYDTM+eH0YCIPo+qARE3NGKp5opTsZQro1hfPB69
iq+x+R083gtR86paNJVM4gopk5LmVehETvRpIhYudddn1vLY4mRZKZamu8eP
QfIBwVR+3Vc+o1+Puy183Zg0hnVjX11SnCHP9oONyEoEGxF8E/a1CEjUnSMY
g8bk5Wt1fmWz+83W8hwiwyhXyx/bFZoxucb1sLxzWClxvmflfY/tWFFPJXV3
rQ8wDronW9RXW1V6AZwgclX7J1iz6J7bQVYxF47PJLx/9hVN7585v1WxgKpC
GqvkHnX7iqeP/36flI5T7TniCzLc1+P3YKIpqOuTaog46dILR5BH2NUzEq9s
H5V5O8Z0be+ekVlri15sredVNV/tvtpfRymxIWcMslKTfamecYXq5mYc/bOf
sflnj/2a/pkL0oXdWbLHg75C/Yrsowx7UKvdlZgCriaM9596bpOxZ25u7djj
6w3eD2rrCiuo15QZa9zHsvk9s0x9Uj+tdss63Dz2Gfk9D03PLjuxAqoe+CCr
POmeVwyY7DJRXu8wjxlUUFWQfYzye2GaOE7eYwjetHq7Qd+R1Z0YUEvJv+t5
lF1m8/xb7jQZY1eef4sF6nU5pWoPvUckEAnmY3stX3sEiUdU4GstXuIau/vi
hMNTpMozU6Oo/ZQtjqVCeJwc+1zS7x2zWKKor9oLzJjtvSg4FZ+8b359bjKW
+xpKdV6z8vuMiPYU6c3G4YFy2NfPlupn3UPOPOLxOwZ0hsJB8HjVvnM+OU74
MHW4o+ZKcdEueMpz5fe9K1cyfvBaUvXTN9BwNa13g6utUF5EMH6/jH8rQGO3
3+Gy4trart+7jsyBoQdbRfoWPUcH6jIG5Waabt9j23WOrbgfQ+pOZbROe5m3
HXsaO86N7WMtfU55l3l67KBT/gm6WaDPcQR50aY4Hk5BIOni1Hh+1+vMeNXu
YWocYy+xgz5+yeO9bWQ8j8u2H49+0nBG6vQuHWFZSnwVBmpTV39YveVGyoYf
370L977SPUKf3smsNNBX2/X028hO6Vd78rmKlusF+vL+4Tb01dYqdEJVs2ds
e6AvGWb+okMuE9dv4rxzZATb3e9GEEzIfa7OW+gOIhb6C+0x7pu4upID19y/
u8Gy6C6UVYoVZXpqiy73bEEp4ku/43R/lEyrAiWtbjUpVaiRWMawUaDUymx8
DU8zal4zPRyj41B4kw5Rj+F5o6O+hIvcBSxcVed41VCP42bluWlg6mv+Cwm4
DdeQjCC1fv8sFa1XVFS4CzJle9tQUeRZ0eBLeslad6IhO1JIA0sDMynf3Ag0
NOMb0PB4UnYHFITkaKRV1lyFi9Gi0vouwT0jVBpo3H2l4UOzvdwb9uftCxHb
9NN4tmM/xbo79tPYsVs/yw8l5J3RPmot19wbgxPt6zEcqp/ro9me70NPb5lV
6KP51n76WM453YcajGlI3Du+HvA83C4y2YMJCXucz58YSx5+qgcFpBVd6Fry
5NYj4b2xL2ytCzXdb2/UFmlySw1AmeIHc18+vbRnRMPvWC0KXYVdnkvuaCs1
nzN7oXktfwWvfDNuF5r5q11LLRXZ9/dAr/iVMXrr3r0jMMobv4uUnvfN0Rp8
v66iyeyWNFmbLutSr3T57pY+E1YK5bs+50yX9nFfE9eR+z6tHL8vld3R+zfs
e2q37j59plU/0t1P1r1p5M4bnLu19xnPx+/T2wtW/D+8z0SrfX9JDJxC/UJc
Ub02w6vTfba4K1i4cXfN7PJa3s7+pHwf7bbcd1+GCvepB1fsq1OHui+7dhr3
aSUOk0RbsZpSw1Ymeuf79AHua3zuvnhL+BDxR58a3GPf9j2OO3pPnq8O713J
7ym07+m7wf6e+PAyg+P6umy5cXgq3Yn/AYe7kBY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 213},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[89]=",
 CellID->279518264]
}, Open  ]],

Cell[TextData[{
 "This is a function definition of a function that integrates between two \
consequent zeros. The zeros of the oscillating function ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"k", "(", "x", ")"}], "=", 
    RowBox[{"sin", "(", 
     RowBox[{"\[Omega]", " ", "x"}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 " are ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"i", 
     FractionBox["1", "\[Omega]"], " ", "\[Pi]"}], ",", " ", 
    RowBox[{"i", "\[Element]", "\[DoubleStruckCapitalN]"}]}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->77529768],

Cell[BoxData[
 RowBox[{
  RowBox[{"psum", "[", 
   RowBox[{"i_", "?", "NumberQ"}], "]"}], ":=", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"k", "[", "x", "]"}], 
     RowBox[{"f", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"i", 
       FractionBox["1", "\[Omega]"], " ", "\[Pi]"}], ",", 
      RowBox[{
       RowBox[{"(", 
        RowBox[{"i", "+", "1"}], ")"}], 
       FractionBox["1", "\[Omega]"], " ", "\[Pi]"}]}], "}"}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[5]:=",
 CellID->391998036],

Cell["\<\
Here is the integral estimate computed by sequence convergence acceleration \
(extrapolation).\
\>", "MathCaption",
 CellID->118868105],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"res", "=", 
  RowBox[{"NSum", "[", 
   RowBox[{
    RowBox[{"psum", "[", "i", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"i", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<AlternatingSigns\>\""}], ",", 
    RowBox[{"\"\<VerifyConvergence\>\"", "->", "False"}]}], "]"}]}]], "Input",\

 CellLabel->"In[6]:=",
 CellID->103722368],

Cell[BoxData["0.04928413882639672`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->44391497]
}, Open  ]],

Cell["Here is the exact integral value.", "MathCaption",
 CellID->250278396],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{
   RowBox[{"Integrate", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"k", "[", "x", "]"}], 
      RowBox[{"f", "[", "x", "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}], "//", 
   "N"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->650138467],

Cell[BoxData["0.04928412771556595`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1266740619]
}, Open  ]],

Cell["The integral estimate is very close to the exact value.", "MathCaption",
 CellID->44304967],

Cell[CellGroupData[{

Cell[BoxData[
 FractionBox[
  RowBox[{"Abs", "[", 
   RowBox[{"exact", "-", "res"}], "]"}], 
  RowBox[{"Abs", "[", "exact", "]"}]]], "Input",
 CellLabel->"In[8]:=",
 CellID->231244670],

Cell[BoxData["2.2544440338778298`*^-7"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->259831867]
}, Open  ]],

Cell[TextData[{
 "Here is another check using ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<ExtrapolatingOscillatory\>\""], "InlineFormula"],
 " strategy."
}], "MathCaption",
 CellID->747790554],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"resEO", "=", 
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"\[Omega]", " ", "x"}], "]"}], 
     RowBox[{"f", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<ExtrapolatingOscillatory\>\""}]}], 
   "]"}]}]], "Input",
 CellLabel->"In[94]:=",
 CellID->831831073],

Cell[BoxData["0.04928411668566339`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[94]=",
 CellID->47233009]
}, Open  ]],

Cell[TextData[{
 "The integral estimate by ",
 Cell[BoxData["\"\<ExtrapolatingOscillatory\>\""], "InlineFormula"],
 " is very close to the exact value."
}], "MathCaption",
 CellID->231675794],

Cell[CellGroupData[{

Cell[BoxData[
 FractionBox[
  RowBox[{"Abs", "[", 
   RowBox[{"exact", "-", "resEO"}], "]"}], 
  RowBox[{"Abs", "[", "exact", "]"}]]], "Input",
 CellLabel->"In[95]:=",
 CellID->3084055],

Cell[BoxData["2.238023289039648`*^-7"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[95]=",
 CellID->667552690]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Double-Exponential Oscillatory Integration", "Section",
 CellID->48895312],

Cell[TextData[{
 "The strategy ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " is for slowly decaying oscillatory integrals over one-dimensional infinite \
regions that have integrands of the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"sin", "(", 
     RowBox[{
      RowBox[{"\[Omega]", " ", 
       SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], 
    RowBox[{"f", "(", "x", ")"}]}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"cos", "(", 
     RowBox[{
      RowBox[{"\[Omega]", " ", 
       SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], 
    RowBox[{"f", "(", "x", ")"}]}], TraditionalForm]], "InlineMath"],
 ", or ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"\[ImaginaryI]", " ", "\[Omega]", " ", 
      SuperscriptBox["x", "p"]}]], 
    RowBox[{"f", "(", "x", ")"}]}], TraditionalForm]], "InlineMath"],
 ", where ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " is integration variable, and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Omega]", ",", " ", "c", ",", " ", "p"}], TraditionalForm]], 
  "InlineMath"],
 " are constants."
}], "Text",
 CellID->395531026],

Cell[TextData[{
 "Integration with ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->467756504],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"2", "*", "x"}], "]"}], "*", 
    RowBox[{"(", 
     RowBox[{"1", "/", 
      SuperscriptBox["x", "2"]}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}], ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<DoubleExponentialOscillatory\>\"", ",", 
      RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->59861109],

Cell[BoxData["0.06333576941856282`"], "Output",
 ImageSize->{67, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->77545093]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["None",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/None"], Cell[TextData[{
     "integration strategy which will be used if ",
     Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
     " fails"
    }], "TableText"]},
   {"\"\<TuningParameters\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "tuning parameters of the error estimation", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic processing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->234406888],

Cell[TextData[{
 "Options of ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 "."
}], "Caption",
 CellID->200724325],

Cell[TextData[{
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " is based on the strategy ",
 Cell[BoxData[
  ButtonBox["\"\<DoubleExponential\>\"",
   BaseStyle->"Link",
   ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#526196975"]], 
  "InlineFormula"],
 ", but instead of using a transformation that reaches double-exponentially \
the ends of the integration interval ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " uses a transformation that reaches double-exponentially the zeros of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"cos", "(", 
    RowBox[{
     RowBox[{"\[Omega]", " ", 
      SuperscriptBox["x", "p"]}], "+", "c"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 ". The theoretical foundations and properties of the algorithm are explained \
in [",
 ButtonBox["OouraMori91",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#1713173531"],
 "], [",
 ButtonBox["OouraMori99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#32080313"],
 "], [",
 ButtonBox["MoriOoura93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#90390145"],
 "]. The implementation of ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " uses the formulas and the integrator design in [",
 ButtonBox["OouraMori99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#32080313"],
 "]."
}], "Text",
 CellID->159085478],

Cell[TextData[{
 "The algorithm of ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " will be explained using the sine integral"
}], "Text",
 CellID->49736474],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["I", "s"], "=", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "\[Infinity]"], 
    RowBox[{"f", " ", 
     RowBox[{"(", "x", ")"}], " ", "sin", " ", 
     RowBox[{"(", 
      RowBox[{"\[Omega]", " ", "x"}], ")"}], 
     RowBox[{
      RowBox[{"\[DifferentialD]", "x"}], "."}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration1",
 CellID->49827302],

Cell["Consider the following transformation ", "Text",
 CellID->86672803],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    FormBox["x",
     TraditionalForm], "=", 
    FormBox[
     FractionBox[
      RowBox[{"M", " ", 
       RowBox[{"\[Phi]", "(", "t", ")"}]}], "\[Omega]"],
     TraditionalForm]}], ",", " ", 
   RowBox[{
    RowBox[{"\[Phi]", " ", 
     RowBox[{"(", "t", ")"}]}], "=", 
    FractionBox["t", 
     RowBox[{"1", "-", 
      SuperscriptBox["\[ExponentialE]", 
       RowBox[{
        RowBox[{
         RowBox[{"-", "2"}], " ", "t"}], "-", 
        RowBox[{"\[Beta]", " ", 
         RowBox[{"(", 
          RowBox[{
           SuperscriptBox["\[ExponentialE]", "t"], "-", "1"}], ")"}]}], "-", 
        RowBox[{"\[Alpha]", " ", 
         RowBox[{"(", 
          RowBox[{"1", "-", 
           SuperscriptBox["\[ExponentialE]", 
            RowBox[{"-", "t"}]]}], ")"}]}]}]]}]]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration2",
 CellID->308447062],

Cell["\<\
where \[Alpha] and \[Beta] are constants satisfying\
\>", "Text",
 CellID->461685658],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"\[Beta]", "=", 
    FormBox[
     RowBox[{"O", "(", "1", ")"}],
     TraditionalForm]}], ",", " ", 
   RowBox[{"\[Alpha]", "=", 
    RowBox[{"o", "(", 
     FractionBox["1", 
      SqrtBox[
       RowBox[{"M", " ", "log", " ", "M"}]]], ")"}]}], ",", 
   RowBox[{
    FormBox[
     RowBox[{
     "0", "\[LessEqual]", "\[Alpha]", "\[LessEqual]", "\[Beta]", 
      "\[LessEqual]", "1"}],
     TraditionalForm], "."}]}], TraditionalForm]], "DisplayMath",
 CellID->209034391],

Cell["\<\
The parameters \[Alpha] and \[Beta] are chosen to satisfy\
\>", "Text",
 CellID->334275650],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"\[Alpha]", "=", 
    RowBox[{"\[Beta]", "/", 
     SqrtBox[
      RowBox[{"1", "+", 
       FractionBox[
        RowBox[{"M", " ", 
         RowBox[{"log", "(", 
          RowBox[{"M", "+", "1"}], ")"}]}], 
        RowBox[{"4", " ", "\[Pi]"}]]}]]}]}], ",", 
   RowBox[{"\[Beta]", "=", 
    FractionBox["1", "4"]}]}], TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration3",
 CellID->156471582],

Cell[TextData[{
 "(taken from [",
 ButtonBox["OouraMori99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#32080313"],
 "])."
}], "Text",
 CellID->452005311],

Cell[TextData[{
 "Transformation (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration2"],
 ") is applied to (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration1"],
 ") to obtain"
}], "Text",
 CellID->436587507],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["I", "s"], "=", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "\[Infinity]"], 
    RowBox[{"f", " ", 
     RowBox[{"(", 
      RowBox[{"M", " ", 
       RowBox[{"(", 
        RowBox[{"\[Phi]", " ", "t"}], ")"}]}], ")"}], " ", "sin", " ", 
     RowBox[{"(", 
      RowBox[{"M", " ", 
       RowBox[{"(", 
        RowBox[{"\[Phi]", " ", "t"}], ")"}]}], ")"}], "M", " ", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox["\[Phi]", "\[Prime]",
         MultilineFunction->None], " ", "t"}], ")"}], "/", "\[Omega]"}], 
     RowBox[{
      RowBox[{"\[DifferentialD]", "t"}], "."}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration4",
 CellID->223094400],

Cell[TextData[{
 "Note that \[Omega] disappeared in the sine term. The trapezoidal formula \
with equal mesh size ",
 Cell[BoxData[
  FormBox["h", TraditionalForm]], "InlineMath"],
 " applied to (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration4"],
 ") gives "
}], "Text",
 CellID->188193280],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"DEO", "(", 
    RowBox[{
     SubscriptBox["I", "s"], ",", "h"}], ")"}], "=", 
   RowBox[{"M", " ", "h", " ", 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"n", "=", 
       RowBox[{"-", "\[Infinity]"}]}], "\[Infinity]"], 
     RowBox[{"f", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"M", " ", 
         RowBox[{"(", 
          RowBox[{"\[Phi]", " ", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"n", " ", "h"}], ")"}], "/", "\[Omega]"}]}], ")"}], " ", 
         "sin", " ", 
         RowBox[{"(", 
          RowBox[{"M", " ", "\[Phi]", " ", 
           RowBox[{"(", 
            RowBox[{"n", " ", "h"}], ")"}]}], ")"}], " ", 
         SuperscriptBox["\[Phi]", "\[Prime]",
          MultilineFunction->None], " ", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"n", " ", "h"}], ")"}], "/", "\[Omega]"}]}], 
        ","}]}]}]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->131047404],

Cell["which is approximated with the truncated series sum", "Text",
 CellID->183709630],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"DEO", "(", 
    RowBox[{
     SubscriptBox["I", "s"], ",", "h", ",", "N"}], ")"}], "=", 
   RowBox[{"M", " ", "h", " ", 
    RowBox[{
     UnderoverscriptBox["\[Sum]", 
      RowBox[{"n", "=", 
       RowBox[{"-", 
        SubscriptBox["N", "-"]}]}], 
      SubscriptBox["N", "+"]], 
     RowBox[{"f", " ", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"M", " ", 
         RowBox[{"(", 
          RowBox[{"\[Phi]", " ", 
           RowBox[{
            RowBox[{"(", 
             RowBox[{"n", " ", "h"}], ")"}], "/", "\[Omega]"}]}], ")"}], " ", 
         "sin", " ", 
         RowBox[{"(", 
          RowBox[{"M", " ", "\[Phi]", " ", 
           RowBox[{"(", 
            RowBox[{"n", " ", "h"}], ")"}]}], ")"}], " ", 
         SuperscriptBox["\[Phi]", "\[Prime]",
          MultilineFunction->None], " ", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"n", " ", "h"}], ")"}], "/", "\[Omega]"}]}], ",", " ", 
        RowBox[{"N", "=", 
         RowBox[{
          SubscriptBox["N", "-"], "+", 
          SubscriptBox["N", "+"], "+", "1."}]}]}]}]}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration5",
 CellID->78008201],

Cell[TextData[{
 Cell[BoxData[
  FormBox["M", TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox["h", TraditionalForm]], "InlineMath"],
 " are chosen to satisfy"
}], "Text",
 CellID->35726533],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"M", " ", "h"}], "=", 
   RowBox[{"\[Pi]", "."}]}], TraditionalForm]], "DisplayMath",
 CellID->19861079],

Cell[TextData[{
 "The integrand decays double-exponentially at large negative ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " as it can be seen from (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration2"],
 "). While the double-exponential transformation, (",
 
 CounterBox["NumberedEquation", "Double-exponentialStrategy3"],
 ") in the section Double-Exponential strategy, makes the integrand also \
decay double-exponentially at large positive ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 ", the transformation (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration2"],
 ") does not decay the integrand at large positive ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 ". Instead it makes the sampling points approach ",
 "double-exponential",
 "ly to the zeros of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", "(", 
    RowBox[{"\[Omega]", " ", "x"}], ")"}], TraditionalForm]], "InlineMath"],
 " at large positive ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 ". Moreover"
}], "Text",
 CellID->371378575],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"sin", "(", 
     RowBox[{"M", " ", 
      RowBox[{"\[Phi]", "(", 
       RowBox[{"n", " ", "h"}], ")"}]}], ")"}], "\[TildeEqual]", 
    RowBox[{"sin", "(", 
     RowBox[{"M", " ", "n", " ", "h"}], ")"}]}], "=", 
   RowBox[{
    RowBox[{"sin", "(", 
     RowBox[{"n", " ", "\[Pi]"}], ")"}], "=", "0."}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->163959352],

Cell[TextData[{
 "As it is explained in [",
 ButtonBox["OouraMori99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#32080313"],
 "], since ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", " ", 
    RowBox[{"(", 
     RowBox[{"\[Omega]", " ", "x"}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 " is linear near any of its zeros, the integrand decreases \
double-exponentially as ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " approaches a zero of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"sin", " ", 
    RowBox[{"(", 
     RowBox[{"\[Omega]", " ", "x"}], ")"}]}], TraditionalForm]], 
  "InlineMath"],
 ". This is the sense in which (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration5"],
 ") is considered a double-exponential formula."
}], "Text",
 CellID->136098023],

Cell["The relative error is assumed to satisfy", "Text",
 CellID->243551589],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["\[CurlyEpsilon]", "M"], "=", 
   RowBox[{
    FractionBox[
     RowBox[{"\[LeftBracketingBar]", 
      RowBox[{
       SubscriptBox["I", "s"], "-", 
       RowBox[{"DEO", "(", 
        RowBox[{
         SubscriptBox["I", "s"], ",", "h", ",", "N"}], ")"}]}], 
      "\[RightBracketingBar]"}], 
     RowBox[{"\[LeftBracketingBar]", 
      SubscriptBox["I", "s"], "\[RightBracketingBar]"}]], "\[TildeEqual]", 
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"-", 
      FractionBox["A", "h"]}]], "\[TildeEqual]", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"-", 
       FractionBox[
        RowBox[{"A", " ", "M"}], "\[Pi]"]}]], "."}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration6",
 CellID->171941890],

Cell[TextData[{
 "In [",
 ButtonBox["OouraMori99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#32080313"],
 "] the suggested value for ",
 Cell[BoxData[
  FormBox["A", TraditionalForm]], "InlineMath"],
 " is 5."
}], "Text",
 CellID->696840013],

Cell[TextData[{
 "Since the ",
 Cell[BoxData[
  FormBox[
   RowBox[{"DEO", "(", 
    RowBox[{
     SubscriptBox["I", "s"], ",", "h", ",", "N"}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " formulas cannot be made progressive, ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " (as proposed in [",
 ButtonBox["OouraMori99",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#32080313"],
 "]) does between 2 and 4 integration estimates with different ",
 Cell[BoxData[
  FormBox["h", TraditionalForm]], "InlineMath"],
 ". If the desired relative error is ",
 Cell[BoxData[
  FormBox["\[CurlyEpsilon]", TraditionalForm]], "InlineMath"],
 " the integration steps are the following:"
}], "Text",
 CellID->313823217],

Cell[TextData[{
 "1. Choose ",
 Cell[BoxData[
  FormBox[
   RowBox[{"M", "=", 
    SubscriptBox["M", "1"]}], TraditionalForm]], "InlineMath"],
 " such that "
}], "Text",
 CellID->73401497],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["M", "1"], "=", 
    RowBox[{"-", 
     FractionBox[
      RowBox[{"\[Pi]", " ", 
       RowBox[{"log", "(", 
        SqrtBox["\[CurlyEpsilon]"], ")"}]}], "A"]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->78892009],

Cell[TextData[{
 "and compute (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration5"],
 ") with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"M", "=", 
    SubscriptBox["M", "1"]}], TraditionalForm]], "InlineMath"],
 ". Let the result be ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", 
    SubscriptBox["M", "1"]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->591389403],

Cell[TextData[{
 "2. Next, set ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["M", "2"], "=", 
    RowBox[{"2", 
     SubscriptBox["M", "1"]}]}], TraditionalForm]], "InlineMath"],
 ", and compute (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration5"],
 ") with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"M", "=", 
    SubscriptBox["M", "2"]}], TraditionalForm]], "InlineMath"],
 ". Let the result be ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", 
    SubscriptBox["M", "2"]], TraditionalForm]], "InlineMath"],
 ". The relative error of the first integration step ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CurlyEpsilon]", 
     SubscriptBox["M", "1"]], "=", 
    FractionBox[
     RowBox[{"\[LeftBracketingBar]", 
      RowBox[{
       SubscriptBox["I", "s"], "-", 
       SubscriptBox["I", 
        SubscriptBox["M", "1"]]}], "\[RightBracketingBar]"}], 
     RowBox[{"\[LeftBracketingBar]", 
      SubscriptBox["I", "s"], "\[RightBracketingBar]"}]]}], TraditionalForm]],
   "InlineMath"],
 " is assumed to be ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CurlyEpsilon]", 
     SubscriptBox["M", "1"]], "\[TildeEqual]", 
    FractionBox[
     RowBox[{"\[LeftBracketingBar]", 
      RowBox[{
       SubscriptBox["I", 
        SubscriptBox["M", "2"]], "-", 
       SubscriptBox["I", 
        SubscriptBox["M", "1"]]}], "\[RightBracketingBar]"}], 
     RowBox[{"\[LeftBracketingBar]", 
      SubscriptBox["I", 
       SubscriptBox["M", "2"]], "\[RightBracketingBar]"}]]}], 
   TraditionalForm]], "InlineMath"],
 ". From (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration6"],
 ") ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[CurlyEpsilon]", 
     SubscriptBox["M", "2"]], "\[TildeEqual]", 
    SuperscriptBox[
     SubscriptBox["\[CurlyEpsilon]", 
      SubscriptBox["M", "1"]], "2"]}], TraditionalForm]], "InlineMath"],
 ", and therefore, if"
}], "Text",
 CellID->406461080],

Cell[BoxData[
 FormBox[
  RowBox[{"\[CurlyEpsilon]", "\[GreaterEqual]", 
   SuperscriptBox[
    RowBox[{"\[Sigma]", "(", 
     FractionBox[
      RowBox[{"\[LeftBracketingBar]", 
       RowBox[{
        SubscriptBox["I", 
         SubscriptBox["M", "2"]], "-", 
        SubscriptBox["I", 
         SubscriptBox["M", "1"]]}], "\[RightBracketingBar]"}], 
      RowBox[{"\[LeftBracketingBar]", 
       SubscriptBox["I", 
        SubscriptBox["M", "2"]], "\[RightBracketingBar]"}]], ")"}], "2"]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration7",
 CellID->103505947],

Cell[TextData[{
 "is satisfied, where ",
 Cell[BoxData[
  FormBox["\[Sigma]", TraditionalForm]], "InlineMath"],
 " is a robustness factor (by default ",
 Cell[BoxData[
  FormBox["10", TraditionalForm]], "InlineMath"],
 ") ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " exits with result ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", 
    SubscriptBox["M", "2"]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->664373460],

Cell[TextData[{
 "3. If (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration7"],
 ") does not hold, compute"
}], "Text",
 CellID->705188804],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["M", "3"], "=", 
   RowBox[{
    SubscriptBox["M", "2"], 
    FractionBox[
     RowBox[{"log", " ", "\[CurlyEpsilon]"}], 
     RowBox[{"log", "(", " ", 
      SuperscriptBox[
       RowBox[{"\[Sigma]", "(", 
        FractionBox[
         RowBox[{"\[LeftBracketingBar]", 
          RowBox[{
           SubscriptBox["I", 
            SubscriptBox["M", "2"]], "-", 
           SubscriptBox["I", 
            SubscriptBox["M", "1"]]}], "\[RightBracketingBar]"}], 
         RowBox[{"\[LeftBracketingBar]", 
          SubscriptBox["I", 
           SubscriptBox["M", "2"]], "\[RightBracketingBar]"}]], ")"}], "2"], 
      ")"}]]}]}], TraditionalForm]], "DisplayMath",
 CellID->320198599],

Cell[TextData[{
 "and compute (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration5"],
 ") with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"M", "=", 
    SubscriptBox["M", "3"]}], TraditionalForm]], "InlineMath"],
 ". If"
}], "Text",
 CellID->59484944],

Cell[BoxData[
 FormBox[
  RowBox[{"\[CurlyEpsilon]", "\[GreaterEqual]", 
   SuperscriptBox[
    RowBox[{"\[Sigma]", "(", 
     FractionBox[
      RowBox[{"\[LeftBracketingBar]", 
       RowBox[{
        SubscriptBox["I", 
         SubscriptBox["M", "3"]], "-", 
        SubscriptBox["I", 
         SubscriptBox["M", "2"]]}], "\[RightBracketingBar]"}], 
      RowBox[{"\[LeftBracketingBar]", 
       SubscriptBox["I", 
        SubscriptBox["M", "3"]], "\[RightBracketingBar]"}]], ")"}], 
    RowBox[{
     SubscriptBox["M", "3"], "/", 
     SubscriptBox["M", "2"]}]]}], TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration8",
 CellID->63173638],

Cell[TextData[{
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " exits with result ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", 
    SubscriptBox["M", "3"]], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->858666489],

Cell[TextData[{
 "4. If (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration8"],
 ") does not hold, compute "
}], "Text",
 CellID->365518043],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["M", "4"], "=", 
   RowBox[{
    SubscriptBox["M", "3"], 
    FractionBox[
     RowBox[{"log", " ", "\[CurlyEpsilon]"}], 
     RowBox[{"log", "(", " ", 
      SuperscriptBox[
       RowBox[{"\[Sigma]", "(", 
        FractionBox[
         RowBox[{"\[LeftBracketingBar]", 
          RowBox[{
           SubscriptBox["I", 
            SubscriptBox["M", "3"]], "-", 
           SubscriptBox["I", 
            SubscriptBox["M", "2"]]}], "\[RightBracketingBar]"}], 
         RowBox[{"\[LeftBracketingBar]", 
          SubscriptBox["I", 
           SubscriptBox["M", "3"]], "\[RightBracketingBar]"}]], ")"}], 
       RowBox[{
        SubscriptBox["M", "3"], "/", 
        SubscriptBox["M", "2"]}]], ")"}]]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->620677816],

Cell[TextData[{
 "and compute (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration5"],
 ") with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"M", "=", 
    SubscriptBox["M", "4"]}], TraditionalForm]], "InlineMath"],
 ". Let the result be ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", 
    SubscriptBox["M", "4"]], TraditionalForm]], "InlineMath"],
 ". If "
}], "Text",
 CellID->116142820],

Cell[BoxData[
 FormBox[
  RowBox[{"\[CurlyEpsilon]", "\[GreaterEqual]", 
   SuperscriptBox[
    RowBox[{"\[Sigma]", "(", 
     FractionBox[
      RowBox[{"\[LeftBracketingBar]", 
       RowBox[{
        SubscriptBox["I", 
         SubscriptBox["M", "4"]], "-", 
        SubscriptBox["I", 
         SubscriptBox["M", "3"]]}], "\[RightBracketingBar]"}], 
      RowBox[{"\[LeftBracketingBar]", 
       SubscriptBox["I", 
        SubscriptBox["M", "4"]], "\[RightBracketingBar]"}]], ")"}], 
    RowBox[{
     SubscriptBox["M", "4"], "/", 
     SubscriptBox["M", "3"]}]]}], TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration9",
 CellID->43583682],

Cell[TextData[{
 "does not hold ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " issues the message ",
 StyleBox["NIntegrate::deoncon", "InlineCode"],
 ". If the value of the ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " method option is ",
 Cell[BoxData[
  ButtonBox["None",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/None"]], "InlineFormula"],
 ", then ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["I", 
    SubscriptBox["M", "4"]], TraditionalForm]], "InlineMath"],
 " is returned. Otherwise ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " will return the result of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " called with the ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " method option."
}], "Text",
 CellID->113978913],

Cell["For the cosine integral", "Text",
 CellID->598441430],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubscriptBox["I", "c"], "=", 
    RowBox[{
     SubsuperscriptBox["\[Integral]", "0", "\[Infinity]"], 
     RowBox[{"f", " ", 
      RowBox[{"(", "x", ")"}], " ", "cos", " ", 
      RowBox[{"(", 
       RowBox[{"\[Omega]", " ", "x"}], ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], ","}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"Double-exponentialOscillatoryIntegration10",
 CellID->317126400],

Cell[TextData[{
 "the transformation corresponding to (",
 
 CounterBox["NumberedEquation", "Double-exponentialOscillatoryIntegration2"],
 ") is"
}], "Text",
 CellID->359299642],

Cell[BoxData[
 FormBox[
  RowBox[{"x", "=", 
   RowBox[{"M", " ", 
    RowBox[{
     RowBox[{"\[Phi]", "(", 
      RowBox[{"t", "-", 
       FractionBox["\[Pi]", 
        RowBox[{"2", "M"}]]}], ")"}], "/", 
     RowBox[{"\[Omega]", "."}]}]}]}], TraditionalForm]], "DisplayMath",
 CellID->684403156],

Cell[CellGroupData[{

Cell["Generalized integrals", "Subsection",
 CellID->188549426],

Cell[TextData[{
 "Here is the symbolic computation of the regularized divergent integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "\[Infinity]"], 
    RowBox[{
     RowBox[{
      RowBox[{"log", "(", "x", ")"}], " ", 
      RowBox[{"sin", "(", "x", ")"}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->544334372],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Limit", "[", 
   RowBox[{
    RowBox[{"Integrate", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Exp", "[", 
        RowBox[{
         RowBox[{"-", "c"}], " ", "x"}], "]"}], 
       RowBox[{"Log", "[", "x", "]"}], 
       RowBox[{"Sin", "[", "x", "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
      RowBox[{"Assumptions", "\[Rule]", 
       RowBox[{"c", ">", "0"}]}]}], "]"}], ",", 
    RowBox[{"c", "\[Rule]", "0"}]}], "]"}]}]], "Input",
 CellLabel->"In[110]:=",
 CellID->125045281],

Cell[BoxData[
 RowBox[{"-", "EulerGamma"}]], "Output",
 ImageSize->{82, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[110]=",
 CellID->405470178]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " computes the nonregularized integral above in a generalized sense."
}], "MathCaption",
 CellID->413549553],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Log", "[", "x", "]"}], 
     RowBox[{"Sin", "[", "x", "]"}]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}], "-", 
  "exact"}]], "Input",
 CellLabel->"In[111]:=",
 CellID->184507651],

Cell[BoxData["4.899747274578203`*^-12"], "Output",
 ImageSize->{94, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[111]=",
 CellID->692171261]
}, Open  ]],

Cell[TextData[{
 "More about the properties of ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " for divergent Fourier type integrals can found in [",
 ButtonBox["MoriOoura93",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#90390145"],
 "]."
}], "Text",
 CellID->122810321]
}, Open  ]],

Cell[CellGroupData[{

Cell["Non-algebraic multiplicand", "Subsection",
 CellID->662597729],

Cell["Symbolic integration of an oscillatory integral.", "MathCaption",
 CellID->624927366],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"20", " ", "x"}], "]"}], 
     RowBox[{"Cos", "[", 
      RowBox[{"18", "x"}], "]"}], " ", 
     FractionBox["1", 
      SqrtBox[
       RowBox[{"x", "+", "1"}]]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[116]:=",
 CellID->27249826],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "12"], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"3", " ", "\[Pi]", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"BesselJ", "[", 
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "2"]}], ",", "2"}], "]"}], "+", 
       RowBox[{"BesselJ", "[", 
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "2"]}], ",", "38"}], "]"}], "-", 
       RowBox[{"BesselJ", "[", 
        RowBox[{
         FractionBox["1", "2"], ",", "2"}], "]"}], "-", 
       RowBox[{"BesselJ", "[", 
        RowBox[{
         FractionBox["1", "2"], ",", "38"}], "]"}]}], ")"}]}], "+", 
    RowBox[{"16", " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{"19", " ", 
        RowBox[{"HypergeometricPFQ", "[", 
         RowBox[{
          RowBox[{"{", "1", "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            FractionBox["5", "4"], ",", 
            FractionBox["7", "4"]}], "}"}], ",", 
          RowBox[{"-", "361"}]}], "]"}]}], "+", 
       RowBox[{"HypergeometricPFQ", "[", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", 
         RowBox[{"{", 
          RowBox[{
           FractionBox["5", "4"], ",", 
           FractionBox["7", "4"]}], "}"}], ",", 
         RowBox[{"-", "1"}]}], "]"}]}], ")"}]}]}], ")"}]}]], "Output",
 ImageSize->{574, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[116]=",
 CellID->9397973]
}, Open  ]],

Cell[TextData[{
 "If the oscillatory kernel is multiplied by non-algebraic function, ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " still gives a good result."
}], "MathCaption",
 CellID->280330184],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Sin", "[", 
      RowBox[{"20", " ", "x"}], "]"}], 
     RowBox[{"Cos", "[", 
      RowBox[{"18", "x"}], "]"}], " ", 
     FractionBox["1", 
      SqrtBox[
       RowBox[{"x", "+", "1"}]]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"PrecisionGoal", "\[Rule]", "10"}]}], "]"}], "-", 
  "exact"}]], "Input",
 CellLabel->"In[117]:=",
 CellID->216018946],

Cell[BoxData[
 RowBox[{"-", "1.920808345712288`*^-9"}]], "Output",
 ImageSize->{97, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[117]=",
 CellID->97367469]
}, Open  ]],

Cell["Plots of the integrand and its oscillatory kernel.", "MathCaption",
 CellID->147553114],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Sin", "[", 
       RowBox[{"20", " ", "x"}], "]"}], 
      RowBox[{"Cos", "[", 
       RowBox[{"18", "x"}], "]"}], " ", 
      FractionBox["1", 
       SqrtBox[
        RowBox[{"x", "+", "1"}]]]}], ",", 
     RowBox[{"Sin", "[", 
      RowBox[{"20", "x"}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[119]:=",
 CellID->189898543],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 105},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[119]=",
 CellID->267888773]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Crude Monte Carlo and Quasi Monte Carlo Strategies", "Section",
 CellTags->"CrudeMonteCarlo",
 CellID->20795380],

Cell["\<\
The crude Monte Carlo algorithm estimates a given integral by averaging \
integrand values over uniformly distributed random points in the integral's \
region. The number of points is incremented until the estimated standard \
deviation is small enough to satisfy the specified precision or accuracy \
goals. A Monte Carlo algorithm is called a quasi Monte Carlo algorithm if it \
uses equidistributed, deterministically generated sequences of points instead \
of uniformly distributed random points.\
\>", "Text",
 CellID->206547588],

Cell["Here is a crude Monte Carlo integration.", "MathCaption",
 CellID->42049581],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x", "4"], "+", 
       SuperscriptBox["y", "4"]}], ")"}]}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<MonteCarlo\>\""}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->127578441],

Cell[BoxData["3.2904253403649304`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->1917618494]
}, Open  ]],

Cell["Here is a crude quasi Monte Carlo integration.", "MathCaption",
 CellID->10760021],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x", "4"], "+", 
       SuperscriptBox["y", "4"]}], ")"}]}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<QuasiMonteCarlo\>\""}]}], "]"}]], "Input",\

 CellLabel->"In[4]:=",
 CellID->836609554],

Cell[BoxData["3.2863210161761502`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->1409774016]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["\"\<MonteCarloRule\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#430697921"], 
    Cell["Monte Carlo rule specification", "TableText"]},
   {"MaxPoints", "50000", Cell[
    "maximum number of sampling points", "TableText"]},
   {"\"\<RandomSeed\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "a seed to reset the random generator", "TableText"]},
   {"\"\<Partitioning\>\"", "1", Cell[
    "partitioning of the integration region along each axis", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", "0", Cell[
    "number of seconds to do symbolic preprocessing ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->36281613],

Cell[TextData[{
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->106038162],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"MaxPoints", "50000", Cell[
    "maximum number of sampling points", "TableText"]},
   {"\"\<Partitioning\>\"", "1", Cell[
    "partitioning of the integration region along each axis", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", "0", Cell[
    "number of seconds to do symbolic preprocessing ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->48148952],

Cell[TextData[{
 Cell[BoxData["\"\<QuasiMonteCarlo\>\""], "InlineFormula"],
 " options."
}], "Caption",
 CellID->943865885],

Cell[TextData[{
 "In Monte Carlo methods [",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "] the ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["\[Integral]", "V"], 
    RowBox[{
     RowBox[{"f", "(", "x", ")"}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " is interpreted as the following expected (mean) value:"
}], "Text",
 CellID->625098730],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["\[Integral]", "V"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}], "=", 
    RowBox[{
     FormBox[
      RowBox[{
       RowBox[{"vol", "(", "V", ")"}], " ", 
       RowBox[{"\[Integral]", 
        RowBox[{
         FractionBox["1", 
          RowBox[{"vol", "(", "V", ")"}]], Cell[BoxData[
          ButtonBox["Boole",
           BaseStyle->"Link",
           ButtonData->"paclet:ref/Boole"]], "InlineFormula"], 
         RowBox[{"(", 
          RowBox[{"x", "\[Element]", "V"}], ")"}], " ", 
         RowBox[{"f", "(", "x", ")"}], 
         RowBox[{"\[DifferentialD]", "x"}]}]}]}],
      TraditionalForm], "=", 
     FormBox[
      RowBox[{
       RowBox[{"vol", "(", "V", ")"}], " ", 
       RowBox[{"E", "(", "f", ")"}]}],
      TraditionalForm]}]}], ","}], TraditionalForm]], "NumberedEquation",
 CellTags->"CrudeMonteCarloAndQuasiMonteCarloStrategies1",
 CellID->178714798],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"E", "(", "f", ")"}], TraditionalForm]], "InlineMath"],
 " is the mean value (the expectation) of the function ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " interpreted as a random variable, with respect to the uniform distribution \
on ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 ", that is, the distribution with probability density ",
 Cell[BoxData[
  RowBox[{
   StyleBox["vol", "TR"], 
   SuperscriptBox[
    RowBox[{"(", "V", ")"}], 
    RowBox[{"-", 
     StyleBox["1", "TR"]}]], 
   ButtonBox["Boole",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Boole"], 
   RowBox[{"(", 
    RowBox[{
     StyleBox["x", "TI"], "\[Element]", "V"}], ")"}]}]], "InlineFormula"],
 ". With ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["Boole",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Boole"], 
   RowBox[{"(", 
    RowBox[{
     StyleBox["x", "TI"], "\[Element]", "V"}], ")"}]}]], "InlineFormula"],
 " is denoted the characteristic function of the region ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 ", with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"vol", "(", "V", ")"}], TraditionalForm]], "InlineMath"],
 " is denoted the volume of ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->133705957],

Cell[TextData[{
 "The crude Monte Carlo estimate is made with the integration rule ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 ". The formulas for the integral and error estimation are given in the \
section \"",
 ButtonBox["MonteCarloRule\"",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#430697921"],
 " in the tutorial ",
 ButtonBox["Integration Rules",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules"],
 ". "
}], "Text",
 CellID->668737273],

Cell["Consider the integral ", "Text",
 CellID->50385805],

Cell[BoxData[
 FormBox[
  RowBox[{
   SubscriptBox["\[Integral]", "\[CapitalOmega]"], 
   RowBox[{
    RowBox[{"f", "(", "x", ")"}], 
    RowBox[{
     RowBox[{"\[DifferentialD]", "x"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->28299325],

Cell[TextData[{
 "If the original integration region ",
 Cell[BoxData[
  FormBox["\[CapitalOmega]", TraditionalForm]], "InlineMath"],
 " is partitioned into the set of disjoint subregions ",
 Cell[BoxData[
  FormBox[
   SubsuperscriptBox[
    RowBox[{"{", 
     SubscriptBox["\[CapitalOmega]", "i"], "}"}], 
    RowBox[{"i", "=", "1"}], "m"], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[CapitalOmega]", "=", 
    RowBox[{
     UnderoverscriptBox["\[Union]", 
      RowBox[{"i", "=", "1"}], "m"], 
     SubscriptBox["\[CapitalOmega]", "i"]}]}], TraditionalForm]], 
  "InlineMath"],
 ", then the integral estimate is "
}], "Text",
 CellID->238424011],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    UnderoverscriptBox["\[Sum]", 
     RowBox[{"i", "=", "1"}], "m"], 
    RowBox[{"MC", "(", 
     RowBox[{"f", ",", 
      SubscriptBox["n", "i"]}], ")"}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->787327054],

Cell["and integration error is ", "Text",
 CellID->83386037],

Cell[BoxData[
 FormBox[
  RowBox[{
   UnderoverscriptBox["\[Sum]", 
    RowBox[{"i", "=", "1"}], "m"], 
   RowBox[{
    RowBox[{"SD", "(", 
     RowBox[{"f", ",", 
      SubscriptBox["n", "i"]}], ")"}], "."}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->42550707],

Cell[TextData[{
 "The number of sampling points used on each subregion generally can be \
different, but in ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s Monte Carlo algorithms all ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["n", "i"], TraditionalForm]], "InlineMath"],
 " are equal (",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["n", "1"], "=", 
    RowBox[{
     SubscriptBox["n", "2"], "=", 
     RowBox[{"\[Ellipsis]", "=", 
      SubscriptBox["n", "m"]}]}]}], TraditionalForm]], "InlineMath"],
 "). "
}], "Text",
 CellID->88279734],

Cell[TextData[{
 "The partitioning ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[CapitalOmega]", "=", 
    RowBox[{
     UnderoverscriptBox["\[Union]", 
      RowBox[{"i", "=", "1"}], "m"], 
     SubscriptBox["\[CapitalOmega]", "i"]}]}], TraditionalForm]], 
  "InlineMath"],
 " is called ",
 "stratification",
 ", and each ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["\[CapitalOmega]", "i"], TraditionalForm]], "InlineMath"],
 " is called ",
 "strata",
 ". Stratification can be used to ",
 ButtonBox["improve crude Monte Carlo estimations",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#473354685"],
 ". (",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 ButtonBox["adaptive Monte Carlo algorithm",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#65285686"],
 " uses recursive stratification.)"
}], "Text",
 CellID->777939290],

Cell[CellGroupData[{

Cell["AccuracyGoal and PrecisionGoal", "Subsection",
 CellID->35773270],

Cell[TextData[{
 "The default values for ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " are ",
 Cell[BoxData[
  ButtonBox["Infinity",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Infinity"]], "InlineFormula"],
 " and 2 respectively when ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s Monte Carlo algorithms are used."
}], "Text",
 CellID->145916005]
}, Open  ]],

Cell[CellGroupData[{

Cell["MaxPoints", "Subsection",
 CellTags->"AMCMaxPoints",
 CellID->1551213],

Cell[TextData[{
 "The option ",
 Cell[BoxData["MaxPoints"], "InlineFormula"],
 " specifies what is the maximum number of (pseudo) random sampling points to \
be used to compute the Monte Carlo estimate of an integral. "
}], "Text",
 CellID->13149194],

Cell[TextData[{
 "Here is an example in which the maximum number of sampling points is \
reached and ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " stops with a message."
}], "MathCaption",
 CellID->57287644],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox["1", 
    SqrtBox["x"]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0.01", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<MonteCarlo\>\"", ",", 
      RowBox[{"\"\<MaxPoints\>\"", "\[Rule]", 
       SuperscriptBox["10", "3"]}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[261]:=",
 CellID->609545809],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"maxp\"\>"}], ":", 
  " ", "\<\"The integral failed to converge after \\!\\(1100\\) integrand \
evaluations. NIntegrate obtained \\!\\(1.768394116870677`\\) and \
\\!\\(0.03357978772002253`\\) for the integral and error estimates.\"\>"}]], \
"Message", "MSG",
 CellLabel->"During evaluation of In[261]:=",
 CellID->821329525],

Cell[BoxData["1.768394116870677`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[261]=",
 CellID->66877318]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"RandomSeed\"", "Subsection",
 CellTags->"AMCRandomSeed",
 CellID->605699968],

Cell[TextData[{
 "The value of the option ",
 Cell[BoxData["\"\<RandomSeed\>\""], "InlineFormula"],
 " is used to seed the random generator used to make the sampling integration \
points. In that respect the use ",
 Cell[BoxData["\"\<RandomSeed\>\""], "InlineFormula"],
 " in Monte Carlo method is similar to the use of ",
 Cell[BoxData[
  ButtonBox["SeedRandom",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SeedRandom"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomReal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomReal"]], "InlineFormula"],
 "."
}], "Text",
 CellID->486128749],

Cell[TextData[{
 "By using ",
 Cell[BoxData["\"\<RandomSeed\>\""], "InlineFormula"],
 " the results of a Monte Carlo integration can be reproduced. The results of \
the following two runs are identical."
}], "Text",
 CellID->705565846],

Cell[TextData[{
 "Here is a Monte Carlo integration that uses ",
 Cell[BoxData["\"\<RandomSeed\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->899418025],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox["x"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0.01", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<MonteCarlo\>\"", ",", 
       RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "12"}]}], "}"}]}]}], "]"}], "//",
   "InputForm"}]], "Input",
 CellLabel->"In[56]:=",
 CellID->10578908],

Cell["1.7828815270494558", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[56]//InputForm=",
 CellID->456655323]
}, Open  ]],

Cell["This Monte Carlo integration gives the same number.", "MathCaption",
 CellID->106384927],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox["1", 
     SqrtBox["x"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0.01", ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<MonteCarlo\>\"", ",", 
       RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "12"}]}], "}"}]}]}], "]"}], "//",
   "InputForm"}]], "Input",
 CellLabel->"In[57]:=",
 CellID->99244934],

Cell["1.7828815270494558", "Output",
 ImageSize->{126, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[57]//InputForm=",
 CellID->254669287]
}, Open  ]],

Cell["\<\
The following shows the first 20 points used in the Monte Carlo integrations.\
\
\>", "MathCaption",
 CellID->698493494],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pnts", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       FractionBox["1", 
        SqrtBox["x"]], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0.01", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<MonteCarlo\>\"", ",", 
          RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "12"}]}], "}"}]}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], 
    "\[LeftDoubleBracket]", 
    RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}], ";"}], "\n", 
 RowBox[{"Take", "[", 
  RowBox[{"pnts", ",", "20"}], "]"}]}], "Input",
 CellLabel->"In[65]:=",
 CellID->71693915],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.1493944534478074`", ",", "0.046079695580606546`", ",", 
   "0.5261966904505617`", ",", "0.40225374288377025`", ",", 
   "0.24985785831144747`", ",", "0.1553506553988744`", ",", 
   "0.7520100494052882`", ",", "0.4476334589492385`", ",", 
   "0.8265965611511349`", ",", "0.8998216037831945`", ",", 
   "0.6722856363929055`", ",", "0.3222493836029982`", ",", 
   "0.7370468685266633`", ",", "0.1626061513547664`", ",", 
   "0.5333903896726376`", ",", "0.12339042082721283`", ",", 
   "0.3674698075757201`", ",", "0.09592096655560528`", ",", 
   "0.8382703076006776`", ",", "0.16101978421091273`"}], "}"}]], "Output",
 ImageSize->{471, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[66]=",
 CellID->163005376]
}, Open  ]],

Cell[TextData[{
 "The points coincide with the points made using ",
 Cell[BoxData[
  ButtonBox["SeedRandom",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SeedRandom"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Random",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Random"]], "InlineFormula"],
 StyleBox[".", "Text"]
}], "MathCaption",
 CellID->204204637],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"SeedRandom", "[", "12", "]"}], ";", 
  RowBox[{"RandomReal", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"0.01", ",", "1"}], "}"}], ",", "20"}], "]"}]}]], "Input",
 CellLabel->"In[67]:=",
 CellID->103533763],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.1493944534478074`", ",", "0.04607969558060655`", ",", 
   "0.5261966904505617`", ",", "0.40225374288377025`", ",", 
   "0.24985785831144747`", ",", "0.1553506553988744`", ",", 
   "0.7520100494052882`", ",", "0.4476334589492385`", ",", 
   "0.8265965611511349`", ",", "0.8998216037831945`", ",", 
   "0.6722856363929055`", ",", "0.3222493836029982`", ",", 
   "0.7370468685266633`", ",", "0.1626061513547664`", ",", 
   "0.5333903896726376`", ",", "0.12339042082721284`", ",", 
   "0.3674698075757201`", ",", "0.0959209665556053`", ",", 
   "0.8382703076006776`", ",", "0.16101978421091273`"}], "}"}]], "Output",
 ImageSize->{471, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[67]=",
 CellID->28224970]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stratified crude Monte Carlo integration", "Subsection",
 CellTags->"StratifiedMonteCarloIntegration",
 CellID->473354685],

Cell["\<\
In stratified sampling Monte Carlo integration you break the region into \
several subregions and apply the crude Monte Carlo estimate on each subregion \
separately.\
\>", "Text",
 CellID->295358603],

Cell[TextData[{
 "From the expected (mean) value formula, Equation (",
 
 CounterBox["NumberedEquation", 
  "CrudeMonteCarloAndQuasiMonteCarloStrategies1"],
 ") at the beginning of ",
 ButtonBox["Crude Monte Carlo and Quasi Monte Carlo Strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#20795380"],
 ", you have"
}], "Text",
 CellID->119054775],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"E", "(", "f", ")"}], "=", 
   RowBox[{
    FractionBox["1", 
     RowBox[{"vol", "(", "V", ")"}]], 
    RowBox[{
     SubscriptBox["\[Integral]", "v"], 
     RowBox[{
      RowBox[{"f", "(", "x", ")"}], 
      RowBox[{
       RowBox[{"\[DifferentialD]", "x"}], "."}]}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"StratifiedCrudeMonteCarloIntegration1",
 CellID->6767063],

Cell[TextData[{
 "Let the region ",
 Cell[BoxData[
  FormBox["V", TraditionalForm]], "InlineMath"],
 " be bisected into two half-regions, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "1"], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "2"], TraditionalForm]], "InlineMath"],
 ". Let ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["E", "i"], "(", "f", ")"}], TraditionalForm]], 
  "InlineMath"],
 " is the expectation of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " on ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "i"], TraditionalForm]], "InlineMath"],
 ", and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["Var", "i"], "(", "f", ")"}], TraditionalForm]], 
  "InlineMath"],
 " is the variance of ",
 Cell[BoxData[
  FormBox["f", TraditionalForm]], "InlineMath"],
 " on ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["V", "i"], TraditionalForm]], "InlineMath"],
 ". From the theorem [",
 ButtonBox["PrFlTeuk92",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#196015578"],
 "]"
}], "Text",
 CellID->252065436],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"Var", "(", "f", ")"}], "=", 
   RowBox[{
    RowBox[{
     FractionBox["1", "4"], " ", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        RowBox[{
         SubscriptBox["E", "1"], "(", "f", ")"}], "-", 
        RowBox[{
         SubscriptBox["E", "2"], "(", "f", ")"}]}], ")"}], "2"]}], "+", 
    RowBox[{
     FractionBox["1", "2"], " ", 
     RowBox[{"(", 
      RowBox[{
       RowBox[{
        SubscriptBox["Var", "1"], "(", "f", ")"}], "+", 
       RowBox[{
        SubscriptBox["Var", "2"], "(", "f", ")"}]}], ")"}]}]}]}], 
  TraditionalForm]], "NumberedEquation",
 CellTags->"StratifiedCrudeMonteCarloIntegration2",
 CellID->83782826],

Cell["\<\
you can see that the stratified sampling gives a variance that is never \
larger than the crude Monte Carlo sampling variance.\
\>", "Text",
 CellID->106756034],

Cell[TextData[{
 "There are two ways to specify strata for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 " strategy. One is to specify \"singular\" points in the variable range \
specifications, the other is to use ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 "'s method sub-option ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 "."
}], "Text",
 CellID->84977076],

Cell["\<\
Stratified crude Monte Carlo integration using variable ranges \
specifications.\
\>", "MathCaption",
 CellID->268168938],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    SuperscriptBox["x", "2"], "+", 
    SuperscriptBox["y", "2"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"2", "/", "3"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", 
     RowBox[{"1", "/", "3"}], ",", 
     RowBox[{"2", "/", "4"}], ",", 
     RowBox[{"3", "/", "4"}], ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<MonteCarlo\>\""}]}], "]"}]], "Input",
 CellLabel->"In[124]:=",
 CellID->1025587076],

Cell[BoxData["0.6663983638680927`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[124]=",
 CellID->35951890]
}, Open  ]],

Cell[TextData[{
 "Stratified crude Monte Carlo integration using ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 "'s sub-option ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->444076593],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    SuperscriptBox["x", "2"], "+", 
    SuperscriptBox["y", "2"]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", " ", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<MonteCarlo\>\"", ",", 
      RowBox[{"\"\<Partitioning\>\"", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"3", ",", "4"}], "}"}]}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[123]:=",
 CellID->53440146],

Cell[BoxData["0.6718523085514982`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[123]=",
 CellID->189281277]
}, Open  ]],

Cell[TextData[{
 "If ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " is given a list of integers, ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["p", "1"], ",", 
     SubscriptBox["p", "2"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["p", "n"]}], "}"}], TraditionalForm]], "InlineMath"],
 " with length ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " that equals the number of integral variables, each dimension ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 " of the integration region is divided into ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["p", "i"], TraditionalForm]], "InlineMath"],
 " equal parts. If ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " is given an integer ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 StyleBox[",", "InlineCode"],
 " all dimensions are divided into ",
 Cell[BoxData[
  FormBox["p", TraditionalForm]], "InlineMath"],
 " equal parts."
}], "Text",
 CellID->260190440],

Cell[TextData[{
 "This graph demonstrates the stratified sampling specified with ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 ". Each cell contains ",
 Cell[BoxData[
  FormBox["3", TraditionalForm]], "InlineMath"],
 " points, as specified by ",
 Cell[BoxData["\"\<MonteCarloRule\>\""], "InlineFormula"],
 "'s option ",
 Cell[BoxData["\"\<Points\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->282679038],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"parts", "=", 
   RowBox[{"{", 
    RowBox[{"3", ",", "4"}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<MonteCarlo\>\"", ",", 
          RowBox[{"\"\<Partitioning\>\"", "\[Rule]", "parts"}], ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<MonteCarloRule\>\"", ",", 
             RowBox[{"\"\<Points\>\"", "\[Rule]", "3"}]}], "}"}]}]}], "}"}]}],
        ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], 
    "\[LeftDoubleBracket]", 
    RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grX", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Line", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"#1", ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"#1", ",", "1"}], "}"}]}], "}"}], "]"}], "&"}], ")"}], "/@",
     
    RowBox[{"Table", "[", 
     RowBox[{"i", ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "0", ",", "1", ",", 
        FractionBox["1", 
         RowBox[{
         "parts", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]]}], 
       "}"}]}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grY", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"Line", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", "#1"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"1", ",", "#1"}], "}"}]}], "}"}], "]"}], "&"}], ")"}], "/@",
     
    RowBox[{"Table", "[", 
     RowBox[{"i", ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "0", ",", "1", ",", 
        FractionBox["1", 
         RowBox[{
         "parts", "\[LeftDoubleBracket]", "2", "\[RightDoubleBracket]"}]]}], 
       "}"}]}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grLP", "=", 
   RowBox[{"Point", "/@", "t"}]}], ";"}], "\n", 
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"grLP", ",", "grX", ",", "grY", ",", "Red", ",", "grLP"}], "}"}], 
   ",", 
   RowBox[{"Axes", "->", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[95]:=",
 CellID->158745299],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXE9T00AUD0lLLbSUURgRKHR01BkQi6AzHoCjnPwQnY4zcHB0gG/Ad/Li
QRi5OB68yZGDH8JjzCZtXvrL23abvNYU0pm22ezm93v7/mz2Zbd93zo7+vCx
dXbcbjUOT1qfj47bp413n068U86UZVnX3vv3kuUdu96R/752Xe87/rntfdpe
A0eVHNVSnbX8w9VoXTGoC14W1jphrWUPqKt7B+qUzdT57a2m+ixQjWUVQbqV
WItpQHsUa1GInVk2aFMHXCfWwtbIjBLxbAVDOUYra8lMe30tYCKzsqOy89oI
Je1y8BI6hrIn12UcK86nl7EIvJJ278b6dOe73vlGnS5Hznel5K7rloOonQeU
K6GzFe/zR9gHbIOyY9Sx9dNh+dvhoQd6ubFRGHBdXKflJMx96ksRhpubm0aj
gb4wLGIwboddDMS27rEoHmMnuIusno30Tnr9Uyh4IH97xw11WE2tR1aSEegt
7EIqvT3p1cvP8/Nf8/Pf2+0FaDfDXRcpl7tsXv/U4UNOr04Ibz0wkm4fynvA
sg9lHiWQJfSzmOxpUEMriKKSnmRQZfSYHEVd9L9lyFHuCoq65x/cip70Q3nh
l2lUqwLKJtTXUrDQKMezUH0aFhr1eBaqT8aSJes1wTozgqhkDXNUboRugtZl
ZMySFe4qSicl8l/ckxWcFSdk6fuMZFiOZ25vxCwYcOJMGp/3IGcdOFZZGSi+
TGTA2bvKE1WDPZabsHluisak3IP7Txwog4md4/k/WroGWt5M5l0RH+5m+phZ
1UCnyZgw5mqgJRnUyR9V5FAu/IGoUql0riyMUcpB86o0qPp5VBpU/bxpF/pS
ErHbLvRFEpX6IoM6Gf5+OzOWWfC+piAqeZ8kKnmfDGqWrJFF7yyBh1Sh/FiQ
BcdeKiNLEaR4k4IVx2Yq86wkVTJWtEsRWM1Qy6CBVyLWL0MPB6E2QIplliWW
y8RWgnFWWAadmMlBcmvkSJV58Jwkowlnd712FNkbrq8WXT6LSZoZcrN3qaxT
EDvia+yqTIZG7RxlMu+JOUpwVXB/pNF/SRCVxnJJVBqtZVDT5s9zc3OdK8eZ
P+co2Ue57/bG1nNBVIotc1RuNA3QKKZkZMySFe4qim71QR2Oah44+Kn06FjU
iWBHIUXcylj6tO72RqQUq8NYKeCieJXgQg4hm8Es3jbwBlxNnwNkfGqM9Wbx
g6vpPIu+fhgWshTPoq/P3oiS71SaXJRttzdycCdhGlSKFElUigxExVEe1095
Fn6VRq/nNeibnkWNbK+htRm6fg24Xx9w9UbPgvtfZkU8EPe/mKFugSxVIxbq
6zAs+BQ6bTRtgSwyqFkeL/JcOkfJ84xxsESfnUs9659w/RjsORqNx+M9qiKI
SvckSVS6J8mgTm4mMGh+t8PWJ2PRz/N22PpkLPp53g5bP2m2HxfKl4ODjs3T
oJBNzVAWwd+esqhfT08vbVvZsH+MLIIMPNqw+loEP5JBzb5H3GYUG/zurRHq
sNlZwEL+qGdRd++X0NoMnfxymD6YZ2ky2fE6oOAekmTZMs+i3x2TLFvmWfT7
YbLk57oZavAVvLh/l5CZSSJy/JfhMnPhbPFcKKU6G/6vyP0GCWW8SnWdSX7Q
778Eov98Yk39A1L4LMY=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{215, 214},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[100]=",
 CellID->78130844]
}, Open  ]],

Cell["\<\
Stratified Monte Carlo sampling can be specified if the integration variable \
ranges are given with intermediate singular points. \
\>", "Text",
 CellID->328782203],

Cell["\<\
Stratified Monte Carlo sampling through specification of intermediate \
singular points. \
\>", "MathCaption",
 CellID->20492263],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"ranges", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", 
       FractionBox["1", "4"], ",", 
       FractionBox["3", "5"], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", 
       FractionBox["2", "5"], ",", 
       FractionBox["3", "4"], ",", "1"}], "}"}]}], "}"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{"1", ",", 
       RowBox[{"Evaluate", "[", 
        RowBox[{"Sequence", "@@", "ranges"}], "]"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<MonteCarlo\>\"", ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<MonteCarloRule\>\"", ",", 
             RowBox[{"\"\<Points\>\"", "\[Rule]", "3"}]}], "}"}]}]}], "}"}]}],
        ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grX", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Line", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"#1", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"#1", ",", "1"}], "}"}]}], "}"}], "]"}], "&"}], "/@", 
    RowBox[{"Rest", "@", 
     RowBox[{"ranges", "[", 
      RowBox[{"[", "1", "]"}], "]"}]}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grY", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Line", "[", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "#1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "#1"}], "}"}]}], "}"}], "]"}], "&"}], "/@", 
    RowBox[{"Rest", "@", 
     RowBox[{"ranges", "[", 
      RowBox[{"[", "2", "]"}], "]"}]}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grLP", "=", 
   RowBox[{"Point", "/@", "t"}]}], ";"}], "\n", 
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"grLP", ",", "grX", ",", "grY", ",", "Red", ",", "grLP"}], "}"}], 
   ",", 
   RowBox[{"Axes", "->", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[18]:=",
 CellID->289215354],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztm89PE0EUx5futjbWmGIUBURJLZqgYIkm4kVJTSQxEP4FmsYEDkaDXDx5
IP4dJF48cePCiRjjBRMNZ8PJP8JjnR/dfcvbaXc687rsyjbp0tnZ/XzfzHtv
OjNLV1vbG6/ftLY3263p5a3Wu43N9vvpl2+32Cl3xHGcE/b+fcNhnzvsk3if
dDrsb/i4wI4FVuXyksuv4Wcd8XEqXFeUdfLl4Fo3qHX4uVt96vh9/FRBUecJ
doMfPahxnCKy7mbkihKiTUSu8CJnopToNVOI60aucHvYjC2aVKrxE+qaJG29
oNd7fT2gYzP3ox8fw7LU11Bb6Mf18PpykH5Q14R1Kf3u53op5IWCok8nQ+f9
jA3fh8sya+V4UkWsI/Ozh+LlzPODu7e31y03UXlFlPf397sU0YPOJXb8GbQc
K+EWx5VxLgf1u7u7JrzM6OEy62vo47henVDWr6+vG7aijnph1PT+vq2YR1ZO
xlgxQ6LSROXnony4vMzu/To761wU5aWlJQLqH89j9/5lt1FSf+zs/KpWv7Xb
RNSVTFA45KxtyCk5ZRAK/85++F+0hJ4yJsow7l4jpMK4S0MtIlsvizKMw5Qq
YPtdQirYGkfFfnnUU0UnurE/etMGibAx1CYaavZyCFOuIN/NEVLBh5RU8KEe
tbvMEC/VTooXo6rXz/33RIahgdcFvG2eoq9N2Hh2jveDKOxPQiO6Ho/zxGLn
dD7gdVF8jPkrb7xboFaCHNFTissRSYUcoaGmacSioVSQnx+j+lFUv9hTRbXK
qCDfqulQ35se3wbwtVoF6s1U0uS1nJIOSr6yzilZo+QxO2yKnGvAd2adkArf
lWbUyCog8swUz9ekLnx7GurGztmHMef1nNBM96N4OfP84DYajbW1Nf6xicor
olyv1+Pm06F1TbGrNafFrtVqBwcHvdj+c7e4ebPpeoyQHYolvIqh8Wiaszyn
JLFbuyDKeP+QhgpjKQ11WmkrjJyUKmD7bUIq2EpDzaM5p+SUJChZm9XL62Gs
fEpIhbGRkgpjIw01Td4Y7IkB/zisGergO9V0KlRPDZJrg6vhhaud05l230AL
a8S1R2pCHmJNvEJuqm1A64uCRk9KZchVtTJYplYeNH/wGlmPij1z3Sgq1FRo
ISUVWkhDPT/fd+PI1w1CKviakgq+pqGmyRvZoGQtxvUo8n9VIROqhFTIBEoq
ZAIN9RyP+1Z7sEFOGO2ypjkn4iiv+MGTEfO5UmGskgUniBFLjoyKD+Pj2pwn
KPLLWr1luw6RqpAZmqqWc3qpCpmTjKrtU4Hk1kB4t7SSoCrEgqGqxv854R1V
M6Usj1rZoFju9nafn0IcP7BmQXTasyD+LFgp8tbwKVOiDP6sEVLBs5RU8LH4
zWopsJ2V5LWrPemqNcYzQQls1aD0s1HSAhstaWmKFEypoci5R0iFyKGkQuTQ
UNPkjeysq8XI3J3df6lW2b0vLDhBpFhyZGx8mpkx5pzpswv5g+/uq4CYJk8v
5K9/ILvvKHWxUvR37oPO4KUu5H+yujBC2Oh+505wy+Xy8fGxjc1HVvfpzNr7
/er/JHR0Rv4Bxs0LQg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{221, 219},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[23]=",
 CellID->82129511]
}, Open  ]],

Cell[TextData[{
 "Stratified sampling improves the efficiency of the crude Monte Carlo \
estimation: if the number of strata is ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 ", the standard deviation of the stratified Monte Carlo estimation is ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 " times less of the standard deviation of the crude Monte Carlo estimation. \
(See the following ",
 ButtonBox["example",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#40464370"],
 ".)"
}], "Text",
 CellID->154735960],

Cell["\<\
The following benchmark shows that stratification speeds up the convergence.\
\>", "MathCaption",
 CellID->15211291],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"n", "=", "10"}], ";", 
  RowBox[{"res", "=", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Timing", "[", 
        RowBox[{"Do", "[", 
         RowBox[{
          RowBox[{"NIntegrate", "[", 
           RowBox[{
            FractionBox[
             RowBox[{
              SuperscriptBox["\[ExponentialE]", "x"], "-", "1"}], 
             RowBox[{"\[ExponentialE]", "-", "1"}]], ",", 
            RowBox[{"{", 
             RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"Method", "\[Rule]", 
             RowBox[{"{", 
              RowBox[{"\"\<MonteCarlo\>\"", ",", 
               RowBox[{"\"\<Partitioning\>\"", "\[Rule]", "#1"}], ",", 
               RowBox[{"\"\<MaxPoints\>\"", "\[Rule]", 
                SuperscriptBox["10", "6"]}]}], "}"}]}], ",", 
            RowBox[{"PrecisionGoal", "\[Rule]", "2"}]}], "]"}], ",", 
          RowBox[{"{", "n", "}"}]}], "]"}], "]"}], "\[LeftDoubleBracket]", 
       "1", "\[RightDoubleBracket]"}], "/", "n"}], "&"}], "/@", 
    RowBox[{"Range", "[", "4", "]"}]}]}], ";"}], "\n", 
 RowBox[{"ColumnForm", "[", "res", "]"}]}], "Input",
 CellLabel->"In[120]:=",
 CellID->150622075],

Cell[BoxData[
 InterpretationBox[GridBox[{
    {"0.011498199999999629`"},
    {"0.003999399999999858`"},
    {"0.00259960000000056`"},
    {"0.0020996999999999253`"}
   },
   BaselinePosition->{Baseline, {1, 1}},
   GridBoxAlignment->{
    "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
     "RowsIndexed" -> {}}],
  ColumnForm[{0.011498199999999629`, 0.003999399999999858, 
   0.00259960000000056, 0.0020996999999999253`}],
  Editable->False]], "Output",
 ImageSize->{67, 60},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[121]=",
 CellID->123311753]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Convergence speedup of the stratified Monte Carlo integration\
\>", "Subsection",
 CellTags->"StratifiedMonteCarloSpeedUp",
 CellID->40464370],

Cell[TextData[{
 "The following example confirms that if the number of strata is ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 ", the standard deviation of the stratified Monte Carlo estimation is ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 " times less than the standard deviation of the crude Monte Carlo \
estimation."
}], "Text",
 CellID->38627054],

Cell[TextData[{
 "Here is a stratified integration definition based on the expected (mean) \
value formula (",
 
 CounterBox["NumberedEquation", 
  "CrudeMonteCarloAndQuasiMonteCarloStrategies1"],
 ")."
}], "MathCaption",
 CellID->40608370],

Cell[BoxData[
 RowBox[{
  RowBox[{"MonteCarloEstimate", "[", 
   RowBox[{"f_", ",", "strata_", ",", "n_"}], "]"}], ":=", 
  RowBox[{
   RowBox[{"(", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"#1", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
       ",", 
       SqrtBox[
        RowBox[{
        "#1", "\[LeftDoubleBracket]", "2", "\[RightDoubleBracket]"}]]}], 
      "}"}], "&"}], ")"}], "[", 
   RowBox[{"Total", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         FractionBox[
          RowBox[{"Mean", "[", 
           RowBox[{"f", "/@", "#1"}], "]"}], "strata"], ",", 
         FractionBox[
          RowBox[{"Variance", "[", 
           RowBox[{"f", "/@", "#1"}], "]"}], 
          FractionBox[
           RowBox[{
            SuperscriptBox["strata", "2"], " ", "n"}], "strata"]]}], "}"}], 
       "&"}], ")"}], "/@", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{"Random", "[", 
        RowBox[{"Real", ",", 
         RowBox[{"{", 
          RowBox[{
           FractionBox[
            RowBox[{"i", "-", "1"}], "strata"], ",", 
           FractionBox["i", "strata"]}], "}"}]}], "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"i", ",", "strata"}], "}"}], ",", 
       RowBox[{"{", 
        FractionBox["n", "strata"], "}"}]}], "]"}]}], "]"}], "]"}]}]], "Input",\

 CellLabel->"In[122]:=",
 CellID->372927484],

Cell["Consider this integral.", "MathCaption",
 CellID->32135533],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"f", "[", "x_", "]"}], ":=", 
  FractionBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", "x"], "-", "1"}], 
   RowBox[{"\[ExponentialE]", "-", "1"}]]}], "\n", 
 RowBox[{"N", "[", 
  RowBox[{
   SubsuperscriptBox["\[Integral]", "0", "1"], 
   RowBox[{
    RowBox[{"f", "[", "x", "]"}], 
    RowBox[{"\[DifferentialD]", "x"}]}]}], "]"}]}], "Input",
 CellLabel->"In[123]:=",
 CellID->306325497],

Cell[BoxData["0.4180232931306735`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[124]=",
 CellID->34443336]
}, Open  ]],

Cell["\<\
Here the integral above is approximated with 1000 points for the number of \
strata running from 1 to 40.\
\>", "MathCaption",
 CellID->396572147],

Cell[BoxData[
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"MonteCarloEstimate", "[", 
      RowBox[{"f", ",", "i", ",", "1000"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"i", ",", "1", ",", "40"}], "}"}]}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[125]:=",
 CellID->43696322],

Cell["\<\
These are the ratios between the standard deviations and the nonstratified, \
crude Monte Carlo estimation.\
\>", "MathCaption",
 CellID->42144463],

Cell[BoxData[
 RowBox[{
  RowBox[{"ratios", "=", 
   RowBox[{
    RowBox[{
     RowBox[{"Transpose", "[", "t", "]"}], "\[LeftDoubleBracket]", "2", 
     "\[RightDoubleBracket]"}], "/", 
    RowBox[{
     RowBox[{"Transpose", "[", "t", "]"}], "\[LeftDoubleBracket]", 
     RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}]}], ";"}]], "Input",
 CellLabel->"In[126]:=",
 CellID->41012652],

Cell[TextData[{
 "Note that ",
 Cell[BoxData[
  RowBox[{
   StyleBox["ratios", "TI"], "\[LeftDoubleBracket]", 
   StyleBox["i", "TI"], "\[RightDoubleBracket]"}]], "InlineFormula"],
 " is the ratio for the Monte Carlo estimation with ",
 Cell[BoxData[
  StyleBox["i", "TI"]], "InlineFormula"],
 " number of strata. This allows you to try a least squares fit of the \
function ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", "x"], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  StyleBox["ratios", "TI"]], "InlineFormula"],
 StyleBox[".", "InlineCode"]
}], "MathCaption",
 CellID->237804523],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"p", "[", "x_", "]"}], ":=", 
  RowBox[{"Evaluate", "@", 
   RowBox[{"Fit", "[", 
    RowBox[{"ratios", ",", 
     RowBox[{"{", 
      FractionBox["1", "x"], "}"}], ",", "x"}], "]"}]}]}], "\n", 
 RowBox[{"p", "[", "x", "]"}]}], "Input",
 CellLabel->"In[127]:=",
 CellID->141165043],

Cell[BoxData[
 FractionBox["1.0075026892406713`", "x"]], "Output",
 ImageSize->{51, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[128]=",
 CellID->574414618]
}, Open  ]],

Cell[TextData[{
 "The fitting of ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", "x"], TraditionalForm]], "InlineMath"],
 " shows a coefficient very close to 1, which is a confirmation of the rule \
of thumb that ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 " number of strata give ",
 Cell[BoxData[
  FormBox["s", TraditionalForm]], "InlineMath"],
 "-times faster convergence. ",
 "This is the plot of the ",
 StyleBox["ratios", "InlineCode"],
 " and the ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", "x"], TraditionalForm]], "InlineMath"],
 " least squares fit."
}], "MathCaption",
 CellID->92041267],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"ratios", ",", 
     RowBox[{"p", "/@", 
      RowBox[{"Range", "[", 
       RowBox[{"Length", "[", "ratios", "]"}], "]"}]}]}], "}"}], ",", 
   RowBox[{"PlotRange", "->", "All"}], ",", 
   RowBox[{"PlotStyle", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "Red", "}"}], ",", 
      RowBox[{"{", "Blue", "}"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[130]:=",
 CellID->211714273],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzFWX9sE1Ucf+tdrwW2UcaAEWV2MGNDhJQEE2IwNiHKjBKaSMIIBjYIsQSn
DS6ymEj/MHFZNE4FHL8rOlMMahVGysKwIxVmg9ANWA4dWAjDQgx2CyYF/jm/
771r79drrx0xNre79+59vp/vz/fe3W1Fc6tvU0tz6+aNzc7lW5v9vs0b33a+
+NZWuMWVIYR64O/NGgRtCVrkr0eS4Ko+I45eULbJYRz5IQs0v6HDvHIXOvii
uXO4KBSv4z9UtBQWKh49MR1qqXdJP/rd3WhnAg4k4BMX6RgKrE/Cgaz4xAVW
i4Hno3BAH068b/E5jzOJj+pLvMcBEo11v8iMnCHaAjRxhK063a+RvihK0Sg+
wAzQzXduGaG6AysTFqKyoX5E1gZH1RDiiEa75QHJtYRs+MR5hDNwHw8JZLyh
Kh5wfgEH2AwnLuA6rPGh6+Vj0fVBfLwf44nXYljU+ODQ2fsSPlWkUsReErHg
NpEaCoxN7gS2r0YE7e7yESQQq1xI9KAoHNRq3v/4EWpVwP0D9S6y9ivZDjg6
4jQKXKb3NFaTyaCFOVtwDJ8jGU2npeixewDs8g0SFt5bN0i9d02+wWHNFTUo
hTWTqDTVRKhKwAZXhrGmbREQT+w7B75GwTgpkZC9fIHp+2J8Ko/2PgCpwNoR
4PHOuwjMTnsKMA6U9vAx6PrqThB7hLDvBK0o8VAcO5JOr2HyLsCnqeLgfVxy
r17CQZw57ODvAcBjOQ2MgfpuIAxvOApcydCAlEqhTUwmpzx7LHKkUAU0Bk7e
63rjMi5XqBskucqu4LJ4+lschPdOpY/FcISnA5BXST4hM3Fyf652Fsl32TVS
R+JE+5kMngFtm8c89TehsYQ/55vTg8uuIw5uPIaRVorMnYkNar4nVP2845oZ
jhF4USzDV03tENvsMN+gDSbZuYfQaJ/WEV21MztqYCrFljlyHOlEVxFRpCFW
c4g9yaTU2TrqnjHq4kdgDIyRQiEGu7MI7ZbsfY0fTLuNnhrsIxmyRX584F1w
1cnf9M/oTrR24wJkcsw1sa/WNL61eT1g2zebVFBw70PXzLsNk/rDKw5IySSt
y4J6SNwFNWJi+S7GXjxeQ+yEBctV/VdT+RFx3Qc4iLykzKNsdar7WPtEZ+EM
ojH127jXJXrsA+KWHVhjOxNbTXjDH/8BKQ6vDsFK1cHETSf9wOspb3Usc3sM
fchETSNsMPMbl/8NSy6wfURweDH6Om8m9TujVZuhhmfHIwvfAS59BvRy+t2d
yfs/9v9r+9jVMJX0YR1un7sXz5Aq3fiTun69yThbSyXph7v/8VadlkQRfcJE
VZA+7Hfuymuwx6NPcyiLvGXB3p+RXBWjyT296DMmRznpd2646J93FDfZqClZ
LqdwK9V3Ge0ogFK42KjJOq6dGqtlDv91/2zyeLyTyTEpxzH5TioYQbsKoGBD
gNUWN9kou47r8wKoxPFbbmEYNxUUtlrWtCbun9+rHTVyeGvPh1tOKSjDu4RV
vS7ltkADj8Mynh66oeLRrJoCq85UfZvOny4GB55hPDI+59t03uaTVa/32Afc
teliwJK1SuxnfJvO71JkBZ2/uw2xt5lETNB5vZuZY0Hnn4Ky5HQoHrA5rDpb
9xRABTfGmupjZiiAAFCLsuSqBDarZOxmfg6ay9TQnRrLHdzcWwAlHr/mEq6Z
oeD5Gx7HzVChljONtWe0KCW3XevO+p76qTAHfn/YV0CDf1F/p7df4TB7r2fv
QUp0ZZYS35tpHwICYSmNhcuxuSddSRy+8qg2KJWQj4XLG4XcosWUNdu9abTT
18ccZWO4uc+QD5ucfbYkmA3GK5IODXpg1/kllcPaUSNHZPvZhlkXzFDKjCuE
al92sm2pKartmb72V37OZ7VSV4U4vLPOhrebWu2pPB/dZRoBJf+FUJCi9O37
RqvpKH4LJni0n8lh0eVrvyHTxcw8ypL48qJ7yu8qlhKrnrLAOzS80j8qS8jX
Tz9cMVnU+1L2ywKsYd4TvkV5ZYrVHFjaF1gWe1QWWExhSWVnJN9XNyrZOLs/
1DaozbhSD8q6VqgeXNarYt+oGcppuZEcGjdDKSuRCqW8AcEPtl/YhA9qrJXr
8vgwPJfi5oE8snTvOphnFLZ92PzzjQbX9DbNj+cbpWsGy6q2BT3tq34tZFVT
7algi6yXJjj707/n5fKokW+oikc6hqi8XsL4FYA3yC+xXRj4/s+Jy9PMliLP
ZZcKO/cQfx0rSbfmGTsYlFCwFGkVQv5SXsRcOaRkznBX87+Fsn8Bqm36Bg==
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 110},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[130]=",
 CellID->308726469]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Global Adaptive Monte Carlo and Quasi Monte Carlo Strategies", "Section",
 CellTags->"AdaptiveMonteCarlo",
 CellID->65285686],

Cell["\<\
The global adaptive Monte Carlo and quasi Monte Carlo strategies recursively \
bisect the subregion with the largest variance estimate into two halves, and \
compute integral and variance estimates for each half. \
\>", "Text",
 CellID->31483592],

Cell["Here is an example of adaptive Monte Carlo integration.", "MathCaption",
 CellID->196813740],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x", "4"], "+", 
       SuperscriptBox["y", "4"]}], ")"}]}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<AdaptiveMonteCarlo\>\""}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->244219843],

Cell[BoxData["3.253102212468609`"], "Output",
 ImageSize->{46, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->140724219]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["MonteCarloRule",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#430697921"], 
    Cell[TextData[{
     StyleBox["MonteCarloRule", "InlineCode"],
     " specification"
    }], "TableText"]},
   {
    ButtonBox["\"\<Partitioning\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#574188911"],
     
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
initial partitioning of the integration region along each axis\
\>", "TableText"]},
   {
    ButtonBox["\"\<BisectionDithering\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#193092152"],
     "0", Cell["\<\
offset from the middle of the region side that is parallel to the bisection \
axis\
\>", "TableText"]},
   {
    ButtonBox["\"\<MaxPoints\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#1551213"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum number of (pseudo-)random sampling points to be used", 
     "TableText"]},
   {
    ButtonBox["\"\<RandomSeed\>\"",
     BaseStyle->"Link",
     ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#605699968"],
     
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
random seed used to generate the (pseudo-)random sampling points\
\>", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->23109506],

Cell["\<\
Adaptive (quasi) Monte Carlo uses crude (quasi) Monte Carlo estimation rule \
on each subregion.\
\>", "Text",
 CellID->31591627],

Cell[TextData[{
 "The process of subregion bisection and subsequent bi-integration is \
expected to reduce the global variance, and it is referred as recursive \
stratified sampling. It is motivated by a theorem that states that if a \
region is partitioned into disjoint subregions the random variable variance \
over the region is greater than or equal to the sum of the random variable \
variances over each subregion. (See ",
 ButtonBox["Stratified Monte Carlo integration",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#473354685"],
 " in the section ",
 ButtonBox["Crude Monte Carlo and Quasi Monte Carlo strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#20795380"],
 ".)"
}], "Text",
 CellID->129628843],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s global adaptive Monte Carlo strategy ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " is similar to ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ". There are some important differences though."
}], "Text",
 CellID->20363041],

Cell[TextData[{
 "(i) ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " does not use singularity flattening, and does not have detectors for slow \
convergence and noisy integration.\n(ii) ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " chooses randomly the bisection dimension. To avoid irregular separation of \
different coordinates a dimension recurs only if other dimensions have been \
chosen for bisection. "
}], "Text",
 CellID->502886302],

Cell[TextData[{
 "(iii) ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " can be tuned to bisect the subregions away from the middle. More at ",
 ButtonBox["\"BisectionDithering\"",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#193092152"],
 "."
}], "Text",
 CellID->96050383],

Cell[CellGroupData[{

Cell["MinRecursion and MaxRecursion", "Subsection",
 CellTags->"AMCMinMaxRecursion",
 CellID->61206936],

Cell[TextData[{
 "The options ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " for the adaptive Monte Carlo methods have the same meaning and \
functionality as they do for ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ". See ",
 ButtonBox["MinRecursion and MaxRecursion",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#280704764"],
 "."
}], "Text",
 CellID->300794538]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"Partitioning\"", "Subsection",
 CellTags->"AMCPartitioning",
 CellID->574188911],

Cell[TextData[{
 "The option ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " of ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " provides initial stratification of the integration. It has the same \
meaning and functionality as ",
 Cell[BoxData["\"\<Partitioning\>\""], "InlineFormula"],
 " of the strategy ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 "."
}], "Text",
 CellID->13134979]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"BisectionDithering\"", "Subsection",
 CellTags->"BisectionDithering",
 CellID->193092152],

Cell[TextData[{
 "When the integrand has some special symmetry that puts significant part it \
in the middle of the region, it is better if the bisection is done a little \
bit away from the middle. The value of the option ",
 Cell[BoxData[
  RowBox[{"\"\<BisectionDithering\>\"", "->", 
   StyleBox["dith", "TI"]}]], "InlineFormula"],
 " specifies that the splitting fraction of the region's splitting dimension \
side should be at ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["1", "2"], "\[PlusMinus]", 
    StyleBox["dith", "TI"]}], TraditionalForm]], "InlineMath"],
 " instead of ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", "2"], TraditionalForm]], "InlineMath"],
 ". The sign of ",
 Cell[BoxData[
  StyleBox["dith", "TI"]], "InlineFormula"],
 " is changed in a random manner. The default value given to ",
 Cell[BoxData["\"\<BisectionDithering\>\""], "InlineFormula"],
 " is ",
 Cell[BoxData[
  FormBox[
   FractionBox["1", "10"], TraditionalForm]], "InlineMath"],
 ". The allowed values for ",
 Cell[BoxData[
  StyleBox["dith", "TI"]], "InlineFormula"],
 " are reals in the interval ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", "4"]}], ",", 
     FractionBox["1", "4"]}], "]"}], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->743381406],

Cell["Consider the function.", "MathCaption",
 CellID->145084776],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x_", ",", "y_"}], "]"}], ":=", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{
     RowBox[{"-", " ", "30"}], 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"x", "-", "5"}], ")"}], "4"], "+", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"y", "-", "5"}], ")"}], "4"]}], ")"}]}]]}], ";"}], "\n", 
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"PlotPoints", "\[Rule]", "30"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], "Input",
 CellLabel->"In[195]:=",
 CellID->249650554],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 260},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[196]=",
 CellID->324691524]
}, Open  ]],

Cell["Consider the integral.", "MathCaption",
 CellID->151969036],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "10"}], "}"}]}], "]"}], "\n", 
 RowBox[{"%", "//", "N"}]}], "Input",
 CellLabel->"In[197]:=",
 CellID->330765961],

Cell[BoxData[
 FractionBox[
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{
     RowBox[{
      RowBox[{"-", "4"}], " ", 
      RowBox[{"Gamma", "[", 
       FractionBox["5", "4"], "]"}]}], "+", 
     RowBox[{"Gamma", "[", 
      RowBox[{
       FractionBox["1", "4"], ",", "18750"}], "]"}]}], ")"}], "2"], 
  RowBox[{"4", " ", 
   SqrtBox["30"]}]]], "Output",
 ImageSize->{225, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[197]=",
 CellID->72887720],

Cell[BoxData["0.5999865728769482`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[198]=",
 CellID->75998188]
}, Open  ]],

Cell[TextData[{
 "The integral is seriously underestimated if no bisection dithering is used \
i.e. ",
 Cell[BoxData["\"\<BisectionDithering\>\""], "InlineFormula"],
 " is given 0."
}], "MathCaption",
 CellID->273972648],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "@", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"f", "[", 
       RowBox[{"x", ",", "y"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
         RowBox[{"\"\<BisectionDithering\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
     "]"}], ",", 
    RowBox[{"{", "20", "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[199]:=",
 CellID->342050614],

Cell[BoxData["0.4038300601866343`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[199]=",
 CellID->69449200]
}, Open  ]],

Cell["\<\
The following picture shows why the integral is underestimated. The black \
points are the integration sampling points. It can be seen that half of the \
peak of the integrand is undersampled.\
\>", "MathCaption",
 CellID->258460776],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{"Reap", "[", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"f", "[", 
       RowBox[{"x", ",", "y"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
         RowBox[{"\"\<BisectionDithering\>\"", "\[Rule]", "0"}], ",", 
         RowBox[{"\"\<RandomSeed\>\"", "\[Rule]", "10"}]}], "}"}]}], ",", 
      RowBox[{"PrecisionGoal", "\[Rule]", "2"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", 
        RowBox[{"{", 
         RowBox[{"x", ",", "y", ",", "0"}], "}"}], "]"}]}]}], "]"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"Print", "[", 
  RowBox[{"\"\<Integral value \>\"", ",", 
   RowBox[{"t", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}], 
  "]"}], "\n", 
 RowBox[{
  RowBox[{"cp", "=", 
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"x", ",", "y"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "10"}], "}"}], ",", 
     RowBox[{"PlotPoints", "\[Rule]", "30"}], ",", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"cp", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
     ",", 
     RowBox[{"PointSize", "[", "0.006", "]"}], ",", 
     RowBox[{"Point", "/@", 
      RowBox[{"t", "\[LeftDoubleBracket]", 
       RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}]}], "}"}], ",", 
   RowBox[{"BoxRatios", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1", ",", "0.4"}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"Axes", "->", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[204]:=",
 CellID->82433483],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Integral value \"\>", "\[InvisibleSpace]", 
   "0.29287614957118363`"}],
  SequenceForm["Integral value ", 0.292876149571184],
  Editable->False]], "Print",
 CellLabel->"During evaluation of (V6.0) In[204]:=",
 CellID->514997194],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 290},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[207]=",
 CellID->441048991]
}, Open  ]],

Cell[TextData[{
 "Specifying bisection dithering of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"10", "%"}], TraditionalForm]], "InlineMath"],
 " gives a satisfactory estimation."
}], "MathCaption",
 CellID->43049328],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "@", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"f", "[", 
       RowBox[{"x", ",", "y"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", " ", 
         RowBox[{"\"\<BisectionDithering\>\"", "->", 
          FractionBox["1", "10"]}]}], "}"}]}]}], "]"}], ",", 
    RowBox[{"{", "30", "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[212]:=",
 CellID->19653156],

Cell[BoxData["0.596771797980303`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[212]=",
 CellID->100562884]
}, Open  ]],

Cell["\<\
It can be seen on this plot, that the peak of the integrand is sampled \
better.\
\>", "MathCaption",
 CellID->56281369],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"t", "=", 
   RowBox[{"Reap", "[", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      RowBox[{"f", "[", 
       RowBox[{"x", ",", "y"}], "]"}], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"y", ",", "0", ",", "10"}], "}"}], ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<AdaptiveMonteCarlo\>\"", ",", 
         RowBox[{"\"\<BisectionDithering\>\"", "\[Rule]", 
          FractionBox["1", "10"]}], ",", 
         RowBox[{"RandomSeed", "\[Rule]", "10"}]}], "}"}]}], ",", 
      RowBox[{"PrecisionGoal", "\[Rule]", "2"}], ",", 
      RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
       RowBox[{"Sow", "[", 
        RowBox[{"{", 
         RowBox[{"x", ",", "y", ",", "0"}], "}"}], "]"}]}]}], "]"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{"Print", "[", 
  RowBox[{"\"\<Integral value \>\"", ",", 
   RowBox[{"t", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}]}], 
  "]"}], "\n", 
 RowBox[{
  RowBox[{"cp", "=", 
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"x", ",", "y"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "10"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", "0", ",", "10"}], "}"}], ",", 
     RowBox[{"PlotPoints", "\[Rule]", "30"}], ",", 
     RowBox[{"PlotRange", "\[Rule]", "All"}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"cp", "\[LeftDoubleBracket]", "1", "\[RightDoubleBracket]"}], 
     ",", 
     RowBox[{"PointSize", "[", "0.006", "]"}], ",", 
     RowBox[{"Point", "/@", 
      RowBox[{"t", "\[LeftDoubleBracket]", 
       RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}]}], "}"}], ",", 
   RowBox[{"BoxRatios", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"1", ",", "1", ",", "0.4"}], "}"}]}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"Axes", "->", "True"}]}], "]"}]}], "Input",
 CellLabel->"In[213]:=",
 CellID->47422202],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"\<\"Integral value \"\>", "\[InvisibleSpace]", 
   "0.6102172250387632`"}],
  SequenceForm["Integral value ", 0.610217225038763],
  Editable->False]], "Print",
 CellLabel->"During evaluation of (V6.0) In[213]:=",
 CellID->230217856],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 290},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[216]=",
 CellID->567881403]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Choice of bisection axis", "Subsection",
 CellTags->"AMCAxis",
 CellID->744911582],

Cell[TextData[{
 "For multidimensional integrals the adaptive Monte Carlo algorithm chooses \
the splitting axis of an integration region in two ways: (i) by random \
selection or (ii) by minimizing the variance of the integral estimates of \
each half. The ",
 ButtonBox["axis selection",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#364207621"],
 " is a responsibility of the ",
 Cell[BoxData[
  ButtonBox["\"\<MonteCarloRule\>\"",
   BaseStyle->"Link",
   ButtonData->"paclet:tutorial/NIntegrateIntegrationRules#430697921"]], 
  "InlineFormula"],
 "."
}], "Text",
 CellID->285991879]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example: Comparison with crude Monte Carlo", "Subsection",
 CellID->718032232],

Cell[TextData[{
 "Generally, the ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " strategy has greater performance than ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 ". This is demonstrated with the examples in this subsection."
}], "Text",
 CellID->368990182],

Cell["Consider the function.", "MathCaption",
 CellID->604440439],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", 
   RowBox[{"x_", ",", "y_"}], "]"}], ":=", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"x", "+", "1"}], ")"}], "2"], "+", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"y", "+", "1"}], ")"}], "2"]}], ")"}]}]], "+", 
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"x", "-", "1"}], ")"}], "2"], "+", 
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"y", "-", "1"}], ")"}], "2"]}], ")"}]}]]}]}]], "Input",
 CellLabel->"In[217]:=",
 CellID->29080614],

Cell["This is a plot of the function.", "MathCaption",
 CellID->170164095],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot3D", "[", 
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x", ",", "y"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[218]:=",
 CellID->51659480],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{360, 278},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[218]=",
 CellID->622651069]
}, Open  ]],

Cell[TextData[{
 "It can be seen from the following profiling that ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " uses nearly three times fewer sampling points than the crude ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 " strategy."
}], "Text",
 CellID->5823547],

Cell[TextData[{
 "These are the sampling points and timing for ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->308769584],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"k", "=", "0"}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"resMC", "=", 
        RowBox[{"NIntegrate", "[", 
         RowBox[{
          RowBox[{"f", "[", 
           RowBox[{"x", ",", "y"}], "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", 
            RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"y", ",", 
            RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
          RowBox[{"Method", "\[Rule]", "\"\<MonteCarlo\>\""}], ",", 
          RowBox[{"PrecisionGoal", "\[Rule]", "2"}], ",", 
          RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
           RowBox[{"k", "++"}]}]}], "]"}]}], ")"}], "//", "Timing"}], "//", 
     "First"}]}], ",", "k"}], "}"}]], "Input",
 CellLabel->"In[219]:=",
 CellID->204184305],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.6898940000000033`", " ", "Second"}], ",", "22500"}], 
  "}"}]], "Output",
 ImageSize->{168, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[219]=",
 CellID->92233666]
}, Open  ]],

Cell[TextData[{
 "These are the sampling points and timing for ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->761215613],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"k", "=", "0"}], ";", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{
      RowBox[{"(", 
       RowBox[{"resAMC", "=", 
        RowBox[{"NIntegrate", "[", 
         RowBox[{
          RowBox[{"f", "[", 
           RowBox[{"x", ",", "y"}], "]"}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", 
            RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"y", ",", 
            RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
          RowBox[{"Method", "\[Rule]", "\"\<AdaptiveMonteCarlo\>\""}], ",", 
          RowBox[{"PrecisionGoal", "\[Rule]", "2"}], ",", 
          RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
           RowBox[{"k", "++"}]}]}], "]"}]}], ")"}], "//", "Timing"}], "//", 
     "First"}]}], ",", "k"}], "}"}]], "Input",
 CellLabel->"In[220]:=",
 CellID->178668460],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"0.18097199999999564`", " ", "Second"}], ",", "5300"}], 
  "}"}]], "Output",
 ImageSize->{159, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[220]=",
 CellID->121741262]
}, Open  ]],

Cell["This is the exact result.", "MathCaption",
 CellID->557763177],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"exact", "=", 
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"x", ",", "y"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}]}], "]"}]}]], "Input",
 CellLabel->"In[221]:=",
 CellID->38794452],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", "\[Pi]", " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"-", 
     RowBox[{"Erf", "[", 
      RowBox[{"1", "-", "\[Pi]"}], "]"}]}], "+", 
    RowBox[{"Erf", "[", 
     RowBox[{"1", "+", "\[Pi]"}], "]"}]}], ")"}], " ", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"Erf", "[", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "\[Pi]"}], "]"}], "+", 
    RowBox[{"Erf", "[", 
     RowBox[{"1", "+", "\[Pi]"}], "]"}]}], ")"}]}]], "Output",
 ImageSize->{352, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[221]=",
 CellID->281561338]
}, Open  ]],

Cell[TextData[{
 "Here is the timing for 100 integrations with ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->152387351],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"tblMC", "=", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"f", "[", 
         RowBox[{"x", ",", "y"}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
        RowBox[{"Method", "\[Rule]", "\"\<MonteCarlo\>\""}], ",", 
        RowBox[{"PrecisionGoal", "\[Rule]", "2"}]}], "]"}], ",", 
      RowBox[{"{", "100", "}"}]}], "]"}]}], ";"}], "//", "Timing"}]], "Input",\

 CellLabel->"In[222]:=",
 CellID->542102388],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"11.884193999999995`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{152, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[222]=",
 CellID->468175873]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 " integration compares well with the exact result. The numbers below show \
the error of the mean of the integral estimates, the mean of the relative \
errors of the integral estimates, and the variance of the integral \
estimates."
}], "MathCaption",
 CellID->436773076],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{
     RowBox[{"Mean", "[", "tblMC", "]"}], "-", "exact"}], "]"}], ",", 
   RowBox[{"Mean", "[", 
    RowBox[{"Abs", "[", 
     FractionBox[
      RowBox[{"tblMC", "-", "exact"}], "exact"], "]"}], "]"}], ",", 
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"tblMC", "-", "exact"}], ")"}], ".", 
     RowBox[{"(", 
      RowBox[{"tblMC", "-", "exact"}], ")"}]}], 
    RowBox[{"Length", "[", "tblMC", "]"}]]}], "}"}]], "Input",
 CellLabel->"In[223]:=",
 CellID->14243719],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.0013799269716345819`", ",", "0.008136628518358476`", ",", 
   "0.004305691886530808`"}], "}"}]], "Output",
 ImageSize->{251, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[223]=",
 CellID->833370161]
}, Open  ]],

Cell[TextData[{
 "Here is the timing for 100 integrations with ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " which is several times faster than ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 " integrations."
}], "MathCaption",
 CellID->187449111],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"tblAMC", "=", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"f", "[", 
         RowBox[{"x", ",", "y"}], "]"}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
        RowBox[{"Method", "\[Rule]", "\"\<AdaptiveMonteCarlo\>\""}], ",", 
        RowBox[{"PrecisionGoal", "\[Rule]", "2"}]}], "]"}], ",", 
      RowBox[{"{", "100", "}"}]}], "]"}]}], ";"}], "//", "Timing"}]], "Input",\

 CellLabel->"In[233]:=",
 CellID->115298842],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"4.213358999999997`", " ", "Second"}], ",", "Null"}], 
  "}"}]], "Output",
 ImageSize->{152, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[233]=",
 CellID->417561380]
}, Open  ]],

Cell[TextData[{
 "The ",
 Cell[BoxData["\"\<AdaptiveMonteCarlo\>\""], "InlineFormula"],
 " integration result compares well with the exact result. The numbers below \
show the error of the mean of the integral estimates, the mean of the \
relative errors of the integral estimates, and the variance of the integral \
estimates."
}], "MathCaption",
 CellID->542299274],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"Abs", "[", 
    RowBox[{
     RowBox[{"Mean", "[", "tblAMC", "]"}], "-", "exact"}], "]"}], ",", 
   RowBox[{"Mean", "[", 
    RowBox[{"Abs", "[", 
     FractionBox[
      RowBox[{"tblAMC", "-", "exact"}], "exact"], "]"}], "]"}], ",", 
   FractionBox[
    RowBox[{
     RowBox[{"(", 
      RowBox[{"tblAMC", "-", "exact"}], ")"}], ".", 
     RowBox[{"(", 
      RowBox[{"tblAMC", "-", "exact"}], ")"}]}], 
    RowBox[{"Length", "[", "tblAMC", "]"}]]}], "}"}]], "Input",
 CellLabel->"In[234]:=",
 CellID->505982188],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0.012998364083917302`", ",", "0.007422121416866435`", ",", 
   "0.0036647861118746473`"}], "}"}]], "Output",
 ImageSize->{244, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[234]=",
 CellID->247214335]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"MultiPeriodic\"", "Section",
 CellID->107447537],

Cell[TextData[{
 "The strategy ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " transforms all integrals into integrals over the unit cube and periodizes \
the integrands to be one-periodic with respect to each integration variable. \
Different periodizing functions (or none) can be applied to different \
variables. ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " works for integrals with dimension less than or equal to twelve. If ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " is given integrals with higher dimension the ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 " strategy is used."
}], "Text",
 CellID->17440224],

Cell[TextData[{
 "Here is an example of integration with ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->547022618],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"-", 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox["x1", "4"], "+", 
       SuperscriptBox["x2", "4"], "+", 
       SuperscriptBox["x3", "4"]}], ")"}]}]], ",", 
   RowBox[{"{", 
    RowBox[{"x1", ",", 
     RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x2", ",", 
     RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"x3", ",", 
     RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", "\"\<MultiPeriodic\>\""}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->250002353],

Cell[BoxData["5.957351006884839`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1924575042]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<Transformation\>\"", "SidiSin", Cell[
    "periodizing transformation applied to the integrand", "TableText"]},
   {"\"\<MinPoints\>\"", "0", Cell[
    "minimal number of sampling points", "TableText"]},
   {"MaxPoints", 
    SuperscriptBox["10", "5"], Cell[
    "maximum number of sampling points", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to be used for symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->23983981],

Cell[TextData[{
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " can be seen as a ",
 "multidimensional",
 " generalization of the strategy ",
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 ". It can also be seen as a quasi Monte Carlo method."
}], "Text",
 CellID->486482988],

Cell[TextData[{
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " uses lattice integration rules; see [",
 ButtonBox["SloanJoe94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#1965527607"],
 "] [",
 ButtonBox["KrUeb98",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#75031785"],
 "]. "
}], "Text",
 CellID->321249722],

Cell[TextData[{
 "Here ",
 StyleBox["integration lattice",
  FontSlant->"Italic"],
 " in ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", "d"], TraditionalForm]], 
  "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "\[Element]", "\[DoubleStruckCapitalN]"}], TraditionalForm]], 
  "InlineMath"],
 ", is understood to be a discrete subset of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalR]", "d"], TraditionalForm]], 
  "InlineMath"],
 " which is closed under addition and subtraction, and which contains ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[DoubleStruckCapitalZ]", "d"], TraditionalForm]], 
  "InlineMath"],
 ". A lattice integration rule [",
 ButtonBox["SloanJoe94",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateReferences#1965527607"],
 "] is a rule of the form "
}], "Text",
 CellID->215520416],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"Q", " ", 
     RowBox[{"f", "(", "x", ")"}]}], "=", 
    RowBox[{
     FractionBox["1", "N"], 
     RowBox[{
      UnderoverscriptBox["\[Sum]", 
       RowBox[{"i", "=", "1"}], "N"], 
      RowBox[{"f", "(", 
       SubscriptBox["x", "i"], ")"}]}]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->633450821],

Cell[TextData[{
 "where ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["x", "1"], ",", 
     SubscriptBox["x", "2"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["x", "N"]}], "}"}], TraditionalForm]], "InlineMath"],
 " are all the points of an integration lattice contained in ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    RowBox[{"[", 
     RowBox[{"0", ",", "1"}], ")"}], "n"], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->213876792],

Cell[TextData[{
 "If ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " is called on, a ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional integral ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 "'s option ",
 Cell[BoxData["\"\<Transformation\>\""], "InlineFormula"],
 " takes a list of one-argument functions ",
 Cell[BoxData[
  FormBox[
   RowBox[{"{", 
    RowBox[{
     SubscriptBox["f", "1"], ",", 
     SubscriptBox["f", "2"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["f", "d"]}], "}"}], TraditionalForm]], "InlineMath"],
 " that is used to transform the corresponding variables. If ",
 Cell[BoxData["\"\<Transformation\>\""], "InlineFormula"],
 " is given a list with length ",
 Cell[BoxData[
  FormBox["l", TraditionalForm]], "InlineMath"],
 " smaller than ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 ", then the last function, ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["f", "l"], TraditionalForm]], "InlineMath"],
 ", is used for the last ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "-", "l"}], TraditionalForm]], "InlineMath"],
 " integration variables. If ",
 Cell[BoxData["\"\<Transformation\>\""], "InlineFormula"],
 " is given a function, that function will be used to transform all the \
variables."
}], "Text",
 CellID->90122638],

Cell[TextData[{
 "Let ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 " is the dimension of the integral. If ",
 Cell[BoxData[
  FormBox[
   RowBox[{"d", "=", "1"}], TraditionalForm]], "InlineMath"],
 " ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " calls ",
 Cell[BoxData["\"\<Trapezoidal\>\""], "InlineFormula"],
 " after applying the periodizing transformation. For dimensions higher than \
",
 Cell[BoxData[
  FormBox["12", TraditionalForm]], "InlineMath"],
 " ",
 Cell[BoxData["\"\<MonteCarlo\>\""], "InlineFormula"],
 " is called without applying periodizing transformations. ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " uses the so- called ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["2", "d"], TraditionalForm]], "InlineMath"],
 " copy rules for ",
 Cell[BoxData[
  FormBox[
   RowBox[{"2", "\[LessEqual]", "d", "\[LessEqual]", "12"}], 
   TraditionalForm]], "InlineMath"],
 ". ",
 "For each ",
 Cell[BoxData[
  FormBox[
   RowBox[{"2", "\[LessEqual]", "d", "\[LessEqual]", "12"}], 
   TraditionalForm]], "InlineMath"],
 " ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " has a set of copy rules that are used to compute a sequence of integral \
estimates. The rules with smaller number of points are used first. If the \
error estimate of a rule satisfies the precision goal, or if the difference \
of two integral estimates in the sequence satisfies the precision goal, the \
integration stops. "
}], "Text",
 CellID->16786571],

Cell[TextData[{
 "Number of points for the ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["2", "d"], TraditionalForm]], "InlineMath"],
 " copy rules in the rule sets for different dimensions."
}], "MathCaption",
 CellID->422139937],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"First", "/@", "#"}], "&"}], ")"}], "/@", 
    RowBox[{"Rest", "[", "NIntegrate`MultiPeriodicDump`copyrules", "]"}]}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{"MapIndexed", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"#1", "*", 
       RowBox[{"2", "^", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"#2", "[", 
           RowBox[{"[", "1", "]"}], "]"}], "+", "1"}], ")"}]}]}], "&"}], ",", 
     "tbl"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"mlen", "=", 
   RowBox[{"Max", "[", 
    RowBox[{"Length", "/@", "tbl"}], "]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"tbl", "=", 
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Join", "[", 
       RowBox[{"#", ",", 
        RowBox[{"Table", "[", 
         RowBox[{"\"\<\>\"", ",", 
          RowBox[{"{", 
           RowBox[{"mlen", "-", 
            RowBox[{"Length", "[", "#", "]"}]}], "}"}]}], "]"}]}], "]"}], 
      "&"}], ",", "tbl"}], "]"}]}], ";"}], "\n", 
 RowBox[{"Style", "[", 
  RowBox[{
   RowBox[{"TableForm", "[", 
    RowBox[{
     RowBox[{"Transpose", "[", "tbl", "]"}], ",", 
     RowBox[{"TableHeadings", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", 
        RowBox[{"Range", "[", 
         RowBox[{"2", ",", 
          RowBox[{
           RowBox[{"Length", "[", "tbl", "]"}], "+", "1"}]}], "]"}]}], 
       "}"}]}]}], "]"}], ",", "Small"}], "]"}]}], "Input",
 CellLabel->"In[3]:=",
 CellID->10332938],

Cell[BoxData[
 StyleBox[
  TagBox[
   TagBox[GridBox[{
      {
       StyleBox["\[Null]",
        ShowStringCharacters->False], 
       TagBox["2",
        HoldForm], 
       TagBox["3",
        HoldForm], 
       TagBox["4",
        HoldForm], 
       TagBox["5",
        HoldForm], 
       TagBox["6",
        HoldForm], 
       TagBox["7",
        HoldForm], 
       TagBox["8",
        HoldForm], 
       TagBox["9",
        HoldForm], 
       TagBox["10",
        HoldForm], 
       TagBox["11",
        HoldForm], 
       TagBox["12",
        HoldForm]},
      {"1", "4996", "4952", "5008", "5024", "5056", "5248", "4864", "5632", 
       "5120", "6144", "12288"},
      {"2", "10012", "9992", "9904", "10016", "10048", "10112", "10496", 
       "9728", "11264", "10240", "20480"},
      {"3", "20012", "20024", "19984", "19808", "20032", "20096", "20224", 
       "20992", "19456", "22528", "45056"},
      {"4", "40028", "40024", "40048", "39968", "39616", "40064", "40192", 
       "40448", "41984", "38912", "77824"},
      {"5", "80044", "80056", "80048", "80096", "79936", "79232", "80128", 
       "80384", "80896", "83968", "167936"},
      {"6", "160036", "160088", "160112", "160096", "160192", "159872", 
       "158464", "160256", "160768", "161792", "323584"},
      {"7", "320084", "320072", "320176", "320224", "320192", "320384", 
       "319744", "316928", "320512", "321536", "643072"},
      {"8", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "640448", 
       "640384", "640768", "639488", "633856", "641024", "1282048"},
      {"9", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", 
       "1280896", "1280768", "1281536", "1278976", "1267712", "2535424"},
      {"10", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "2561792", "2561536", "2563072", "2557952", "5115904"},
      {"11", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "5123584", "5123072", "5126144", "10252288"},
      {"12", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "10247168", "10246144", "20492288"},
      {"13", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", \
"\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "\<\"\"\>", "20494336", "40988672"}
     },
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxDividers->{
      "Columns" -> {False, True, {False}, False}, "ColumnsIndexed" -> {}, 
       "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.27999999999999997`], {
          Offset[0.7]}, 
         Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}],
    {OutputFormsDump`HeadedRows, OutputFormsDump`HeadedColumns}],
   Function[BoxForm`e$, 
    TableForm[
    BoxForm`e$, 
     TableHeadings -> {Automatic, {2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12}}]]],
  StripOnInput->False,
  FontSize->Small]], "Output",
 ImageSize->{484, 167},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->1821269818]
}, Open  ]],

Cell[CellGroupData[{

Cell["Comparison with \"MultiDimensionalRule\"", "Subsection",
 CellID->161179452],

Cell[TextData[{
 "Generally ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " is slower than ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " using ",
 Cell[BoxData["\"\<MultiDimensionalRule\>\""], "InlineFormula"],
 ". For computing high-dimension integrals with lower precision ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " might give results faster."
}], "Text",
 CellID->549185201],

Cell["This defines function of eight arguments.", "MathCaption",
 CellID->181396511],

Cell[BoxData[
 RowBox[{
  RowBox[{"f", "[", 
   RowBox[{
   "x1_", ",", "x2_", ",", "x3_", ",", "x4_", ",", "x5_", ",", "x6_", ",", 
    "x7_", ",", "x8_"}], "]"}], ":=", 
  RowBox[{"1", "/", 
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{"0.9671190054385935`", " ", "x1"}], "+", 
      RowBox[{"0.21216802639809276`", " ", "x2"}], "+", 
      RowBox[{"0.682779542171783`", " ", "x3"}], "+", 
      RowBox[{"0.32962509624641606`", " ", "x4"}], "+", 
      RowBox[{"0.5549215440908636`", " ", "x5"}], "+", 
      RowBox[{"0.7907543870000786`", " ", "x6"}], "+", 
      RowBox[{"0.8580353669569777`", " ", "x7"}], "+", 
      RowBox[{"0.4796298578498076`", " ", "x8"}]}], ")"}], "9"]}]}]], "Input",\

 CellLabel->"In[8]:=",
 CellID->357870496],

Cell[TextData[{
 "Timings in seconds for computing ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{"\[Ellipsis]", 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{
         SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
         SubscriptBox["x", "8"]}], "]"}], 
       RowBox[{"\[DifferentialD]", 
        SubscriptBox["x", "1"]}], "\[Ellipsis]", 
       RowBox[{"\[DifferentialD]", 
        SubscriptBox["x", "8"]}]}]}]}]}], TraditionalForm]], "InlineMath"],
 " using ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " with ",
 Cell[BoxData["\"\<MultiDimensionalRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->53890916],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"\[IndentingNewLine]", 
  RowBox[{
   RowBox[{
    RowBox[{"tbl", "=", 
     RowBox[{"Table", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"{", 
         RowBox[{
         "\"\<IntegralEstimate\>\"", ",", "\"\<Evaluations\>\"", ",", 
          "\"\<Timing\>\""}], "}"}], "/.", 
        RowBox[{"NIntegrateProfile", "[", 
         RowBox[{
          RowBox[{"NIntegrate", "[", 
           RowBox[{
            RowBox[{"f", "[", 
             RowBox[{
             "x1", ",", "x2", ",", "x3", ",", "x4", ",", "x5", ",", "x6", ",",
               "x7", ",", "x8"}], "]"}], ",", 
            RowBox[{"{", 
             RowBox[{"x1", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x2", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x3", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x4", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x5", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x6", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x7", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"{", 
             RowBox[{"x8", ",", "0", ",", "1"}], "}"}], ",", 
            RowBox[{"Method", "\[Rule]", "meth"}], ",", 
            RowBox[{"MaxPoints", "\[Rule]", 
             SuperscriptBox["10", "8"]}], ",", 
            RowBox[{"PrecisionGoal", "\[Rule]", "pg"}]}], "]"}], ",", "1"}], 
         "]"}]}], ",", 
       RowBox[{"{", 
        RowBox[{"pg", ",", "1", ",", "4"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"meth", ",", 
         RowBox[{"{", 
          RowBox[{"\"\<MultiPeriodic\>\"", ",", 
           RowBox[{"{", 
            RowBox[{"\"\<MultiDimensionalRule\>\"", ",", " ", 
             RowBox[{"\"\<Generators\>\"", "\[Rule]", "5"}]}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"\"\<MultiDimensionalRule\>\"", ",", " ", 
             RowBox[{"\"\<Generators\>\"", "\[Rule]", "9"}]}], "}"}]}], 
          "}"}]}], "}"}]}], "]"}]}], ";"}], "\n", 
   RowBox[{"TableForm", "[", 
    RowBox[{
     RowBox[{"Map", "[", 
      RowBox[{
       RowBox[{
        RowBox[{"#", "[", 
         RowBox[{"[", "3", "]"}], "]"}], "&"}], ",", "tbl", ",", 
       RowBox[{"{", "2", "}"}]}], "]"}], ",", 
     RowBox[{"TableHeadings", "\[Rule]", 
      RowBox[{"Map", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Style", "[", 
          RowBox[{"#", ",", 
           RowBox[{"FontFamily", "\[Rule]", "Times"}], ",", 
           RowBox[{"FontSize", "\[Rule]", "11"}]}], "]"}], "&"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
           "\"\<Precision goal \[Rule] 1\>\"", ",", 
            "\"\<Precision goal \[Rule] 2\>\"", ",", 
            "\"\<Precision goal \[Rule] 3\>\"", ",", 
            "\"\<Precision goal \[Rule] 4\>\""}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"\"\<MultiPeriodic\>\"", ",", 
            RowBox[{"ColumnForm", "[", 
             RowBox[{"{", 
              RowBox[{
              "\"\<MultiDimensionalRule\>\"", ",", " ", 
               "\"\<with 5 generators\>\""}], "}"}], "]"}], ",", 
            RowBox[{"ColumnForm", "[", 
             RowBox[{"{", 
              RowBox[{
              "\"\<MultiDimensionalRule\>\"", ",", " ", 
               "\"\<with 9 generators\>\""}], "}"}], "]"}]}], "}"}]}], "}"}], 
        ",", 
        RowBox[{"{", 
         RowBox[{"-", "1"}], "}"}]}], "]"}]}], ",", 
     RowBox[{"TableSpacing", "\[Rule]", "3"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[11]:=",
 CellID->391777809],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXEtvHEUQbu/sy7txYhxh1uZh85LgkgASJLE5kCAlMYgDEjdOVoQURyBQ
yBH/9mZ7Znqnpurr18zs2ruxJe/sdFdXV1dVV9ej7V/O37z846/zNxcvzo+f
vT7/5+XFi3+Pn/79et6UbSm19Z9S6nKm5t/1/Fv+a34u57/F1+Lb/flnNu/8
0rxlFkrnj6Mc4p75HCx68jalvnDCm8Z+ibXHniorh6sCbtHer9od0POvfdJn
3kfz55jRMWLvA/aeleP75NmbPweg3c5jnh+Sdxc+g2fI+i29Fv/j+XNA8Npx
fQZH+w38g4IVJUcWK+tXM5VMZLRJyfUI1ILJNTw9Ldcg3kcML8VewzZhozmH
uIR6bqpFa0a4FzsGt3Iq3o/hgZMeuEYITaF+SJpjFIUz9O7QKwLRV/V+rmW9
RjplV+GWUeYYNSK7gvOZv3Nap4Iuzg8+gr/vzz+GTo3j2Ljuyz3bTAd2ktfB
OVmXe85zoluFchH0RC5qKDvobIgPPaZDmQOPGCfWyTkqNY2fAB3pRAWhi5No
BiUTt9P5Knq6Ot3k6CeIKwJq4sVhz0++ern/qXUvNIP0qepkqumOLvWGtWWO
vkFJTf3ct9rHueU70xfKyuRlz3TzvKXlGbuYR3DqQOgD27Ul7TtRkjaQP7I2
y8EJxGB1AVMXKT8OQd7vBLjX+X4BEDNGw1CMeAfjqLU8ZVix5m87uSxoXgK3
dzU5rRt6OZ4VtGo9ZFJwWw5huxL8MzqW74SU9UJPJlJiA6tRHr3+SBeSusva
QxHEVOPIwT4/0JWnMyDt27oeCdAIgUYkU+2ODAzco3yFvzMqzcfRTc9SejYt
Zu6XcNxGcU1zxaYhn3lWwtFYmOOjO+G5Ay+N9iw++rT4vtPcSq9rrIw8m/gI
EZ9pyJ6lRECH7B3F793E5GcBSvzrjOMV93a+1VB3mEd33eJhtKObnrMog+WT
N5/rgL1zfFM3LQTqWWBWDh/SavM+FLYh5y3ZMesT9zazDG7PCsWGqZK2eHYQ
LR6NCMmarz1kofDa1zHSHWocPYQjyLA9TJWuO+tggLhM7Qp/Csw6BeNCp0t9
Z6THZHuBlbePwjjGUEw90zEUpNoAvq+gJU7Yp/xUS+f8bbguOE+n/G8aFcfL
wuM/Bv2alPXGrYyfS+lRsIHLPJYHRb3INz/QMpq12S06/oyNn2gce1iLWKzo
6mPBt6dnE6NeVEPiUe/IQVGm69EwryDf1ViDaWWZV4IpvlNd30mZB49936xK
scwjx50UllNc4nE1XE8Fx5vTvuOlpX003KaafYI4C7y1FCrjKl3rGU+H6sru
E3sIeBCXeQhXTbCGpfhFcTn0OHofIfrJewpdQ9h6QthQ/KxnrN7sxksb/8+e
gbKu1r1OtfG1LZ2nDEcbXdqkSnezvF6uf8Hz3voOWKKWOoNmzzlLppiuMdpo
P54rRvJWmm6K+Y6SFYDl5ARoBN12L8i9+i6Ei6tnjwU2eWZiqxu1F3l/wzNE
egfLzCB0Ka248xXvnDb17TbnekiybXIavpxrl3mHHV3ZJt+9Xdu+p8NVdNR+
SvD3ynnROPt8mK/06qP0t6dn0/IRVgNDVXieYUNWjGr0IRsfuunOd8Rp4nib
L7F0fK25jV/XfAT3MtKyEXHQsEoTtPf73pm7vx0fd/v5wYqpCt+uvkeaiS7S
lal6//pmI0KVNRiVermHtSzuFkGoZht6x7rEK83hHIqozAE9qmtJLiGit+uT
d2hmr7jehOrGBov73jm+JZ+gM+LetSvebKUhYpa4+EXak+ufU+jGmiScCDXe
miEHENad76J+eGpUZOY7ca6I/lWAP0+B8//YvwnFsLudSCAyf7xcix4zn8D0
PWzFf08Rp2chG3tVMlpWfSfFqi6LBs5jvMvich1x/n+clFOzDahSSG85+G4q
W9txSOBRbISeZqOcgHE825mp6taDtVFf5Su9+hjc3WMEdGmEdA1oWW7PK13I
6M/y+QRCXWik0a9g62OIIb9JJ6KMbLEx8h8tvQurM9ZrWIwaVDCVfyJbrRfS
Z3h9GZJPWtBq71ysitZZJLcdp1utJerGAbdzwNeXsZj09j222jPjUaN1WfsD
K5dXup68RiKkd0boqLQDzRqWIY3dQWRQ4u5L7JpE9REcN3Mct6Db9Vewq6D7
vaAUqhldmHkcDP9i0oWNx9P9EIT0Xt36xyn7uNW66Bl/PdaDLeBzC1mz2DLu
pKsuWpAmIouMYt4e6LNr5FbcBXfUmHIcXQxWTD/eTb9B2sYljs8Z9EMIjf9n
yKds7GeJ84uxS6XjV/Zu99h2yVUOb/vHZf/PEKv5OFqPHmGpa9nDUq9GQv9t
phP7U8w3U1Wdqt66nTTnuAQcih63/8bzroFKWw3rRDkyZhu7ji2oMQ7vqhMP
DN41cNTUKOwCFzjz5P+zaUlvRP7ORYujDuLKyDvrPiplFV4sN9zxYSm2A9wB
VR6efkbNIqmn8XMn+t4pdQ7/z8s57OVTL0Zk2/xVzmaaWU5hQCY3fIF88Vj5
ttyifnr6uiN0nXNjvCIZ01otnMcbYzWq2vq1Kghxw6kUTm3598UY8k3jqK2G
G/U74+gF/6BX56r2mrZJ91RyHgwq+OKLjIrpGnwxbWxt2rYh24Jn5royveEL
5EvPqev0doq8941vee8zPLtxeDyZ2PtiBF8rzx7zu7yrmNPtLa6QiwLPN8vi
XQczEY6JzAOvT1J55Yqtiv++SPGNNNbxWwk4eT2X//8EK4dabUmpxdy3VzhX
jjMHLv7jttr6H3PQZLk=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{380, 125},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[12]//TableForm=",
 CellID->203263888]
}, Open  ]],

Cell[TextData[{
 "Number of integrand evaluations for computing ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{"\[Ellipsis]", 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", "1"], 
      RowBox[{
       RowBox[{"f", "[", 
        RowBox[{
         SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
         SubscriptBox["x", "8"]}], "]"}], 
       RowBox[{"\[DifferentialD]", 
        SubscriptBox["x", "1"]}], "\[Ellipsis]", 
       RowBox[{"\[DifferentialD]", 
        SubscriptBox["x", "8"]}]}]}]}]}], TraditionalForm]], "InlineMath"],
 " using ",
 Cell[BoxData["\"\<MultiPeriodic\>\""], "InlineFormula"],
 " and ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " with ",
 Cell[BoxData["\"\<MultiDimensionalRule\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->376649596],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"TableForm", "[", 
  RowBox[{
   RowBox[{"Map", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"#", "[", 
       RowBox[{"[", "2", "]"}], "]"}], "&"}], ",", "tbl", ",", 
     RowBox[{"{", "2", "}"}]}], "]"}], ",", 
   RowBox[{"TableHeadings", "\[Rule]", 
    RowBox[{"Map", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"Style", "[", 
        RowBox[{"#", ",", 
         RowBox[{"FontFamily", "\[Rule]", "Times"}], ",", 
         RowBox[{"FontSize", "\[Rule]", "11"}]}], "]"}], "&"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
         "\"\<Precision goal \[Rule] 1\>\"", ",", 
          "\"\<Precision goal \[Rule] 2\>\"", ",", 
          "\"\<Precision goal \[Rule] 3\>\"", ",", 
          "\"\<Precision goal \[Rule] 4\>\""}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"\"\<MultiPeriodic\>\"", ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<MultiDimensionalRule\>\"", ",", " ", 
             "\"\<with 5 generators\>\""}], "}"}], "]"}], ",", 
          RowBox[{"ColumnForm", "[", 
           RowBox[{"{", 
            RowBox[{
            "\"\<MultiDimensionalRule\>\"", ",", " ", 
             "\"\<with 9 generators\>\""}], "}"}], "]"}]}], "}"}]}], "}"}], 
      ",", 
      RowBox[{"{", 
       RowBox[{"-", "1"}], "}"}]}], "]"}]}], ",", 
   RowBox[{"TableSpacing", "\[Rule]", "3"}]}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->554131220],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXOtvHDUQd2739pLQlvAK+UIvlPAStKE8K5SCCggKtCpItP3Ap6hCaioQ
qPQj+dvNeXdnd3b88+t2k9ylFym3Z3s8npfHM+NN7h4+e/zHX4fPjh4d7v7w
9PCfx0eP/t39/u+ns65sTam1/5RSxztq9l3PvpW/5ud49lt9rb7tzz6z2eD7
ppURlC4f0xLimvkcNyNln1LvOeFNZ15jHYmnyurpqoJr+vO23wE9+5qzMdOe
zJ7rgo6JaI9F28wf1wvSU8JzDuRzu35KvGNHf1b3FwJPVq//Wf10rVuIJ/H/
aSWaWkINp3m7ss1bBjU5YlCN0Dt4RtqWodWeCLwcewfbpphdBDQmucghFxJK
Yh25eY/sfTVq3Tjq8Ap4rgeWYfwiio4UfjGGUq4dfWZi5eusm1kmp011xy3p
zGWVBmLk5TFzzJowfyDtBvkHy85yvlpB1PaS/g6gM7yXx7BXzg3rUEpJWpf0
unkcRsh7oWyv2bWhErJummfl5NliXJKFPcB1iOQxEvaYOfBAnry+zbZaeT6F
/FacRvt7t+fH3hw+zevB+AlX2SMbq0FlxGLmjEBf5hgzsiw9YycWIpuX/Pni
nGaLeOwsNG7btoxQsEVs1/RKmVL8hs7lrKaX+MyV7RGkRRQ1vIyXPBZp2QuO
o0K2+GKipKHtOmy/Uf+ge9m3YkOhdzfZO1hyBXdspL8Ix0wK0JSuuS1d2UxI
Y/PGYcNoD8eZGBZlG/3iOynl0D6Mw4Lp6qN3M+0lRAtrX9aVvl8J8EQ+1tgF
z83Gdb8rN9sW8DxHGzMfQ/1jBm/w3hDrGrw5mEe+jnzjJ6UEfhdcmI/pauRE
Rs5blYQsTXrCTNs7AFk+PS874GS1I5TTHzjw8J2FogaCPz/VkKi47AyqH1NE
15zxdRwPsv1VDwri6IrLZpazvhGKQqGORLsgj2HHit7o8Y0ozUkK3HWUkDe5
KdqO+NbCK/diaJ3zUqVIrUlAj2dJ0yH1oSxljux/ILvxSEJK0oyjrHcZKwwU
hSJroSjZOl870pNxLJ3yuwAj1QAox8e1iFC1VmqUUxC2jwJCCb70vHWElwO0
h3ad9Be2tWL6p4F15UkRV58IaeIAreLdW1AeEV7882RNXIriKD1qcUsrLj+e
Cqi4bH/4isTwkVcKhv75v4HLgOcin0oejGqQ/C6XspCs1seo3mm8vxDzKGuh
OsBNbef5Mjvi9QLyf6s8f5Xn98/zXfkzqlBJy3xHtHklC+G7w+D5Dgi9HXFT
S5+9rPm7HauETsb+HhL37iXAxr21INv34KxwlB93kmKoA4aSWQqnUnXHFyH7
tnLbgGQhfGIudcVJqSNXk3vL0/4FYomrp6A3p3j7S8ZO9bOcWXXqe02LrvO7
us3afHxgX5Km9UbCC50Rh2pq6MSiOJTnwfgk29PdLJnfBvn0dCdK/og23/28
fUL3zW/hec7aG1paO85n9xDeIP+oUhOWYtRZLi3EwvJ1sizjMtSQROPq29ga
346CSs9Q3bJOiXXi1pWcVjY9VCZJGSBVrjJlZ5Dvinah2zie4nGD557G8Thl
kBzezK+iorPPn56fkfOWKZKlovfK+IryNPpQ27ULsnyO576Akze91P5OS9+4
rJkgrCb3jO8k5x+4OZ/j/SXSU+xceG8U9Lq3GGtMwxyP6o4vQwZH96PWqe5/
r2729SpYcd5z734UrtS4tauxkhq1jBlZaEf2f0sC9+73WEfS/CAKF8Uhrqje
3oOLn22FdiD2nnTmNZlrB+IjIE2aEdbFQwgVV+1y33XgUzCUIWwFpCN3nYcq
BxYzfh3KOO3MehickSoxPvesJAdzLTbragJPKb0p53WfW7dKrqm5Et2WSclw
j8JzJdN/TePch27VHmgciXJ4Y17flhSffabgHjGCPjbCXgBaTnbkia5082f9
/AZCHWlkiU9g7y2I4SctfQvIlLQ7S6ITqpk1bmHas9DupRMvF3h9edyVHrTS
bfJp0boTKW3HaeD9y5FQFuaK/+y80I4A40452Z7OxRd/C2Cx+HlNI+39yOho
rQOtGtYhz61ym84ad25j1yzripC4WWO3B92ud9dOg+7Xg1poV3RhlrkRrGK7
sMkcKw9B2PGy2/4kZW/24ovO8Hxh+MEe8DZBdjy2neNwrqseZInII6P8agTG
iEfpxV1w07kpd9+vnCb9eDf9Bmlbr3HIO4wbEHoCe98Sc/cS17fmnigdv4o2
7bGNWqoSnsbpzbafIVbzMV2OEctTdypKtV1NLPun/0WA4ykRm6m2mt7t3Uha
c70GLKwRd/wma3GB+4AO1k3lqM6cWz7WoMU4oqtBIjB4p+q4SeCwDS5w5l0c
mt6IN7dctNixqrdK66zLqxQuvFhW0vFhqbYD3AFtzZd/Rq1iU8/z50HsfVDq
HPGfV3I4yudRTOIt1HyWWS9hQDZXcoFy8Xj5vtLicXo63xG2LqWxfko65vd3
cB1vjjXXTZ7fqoIQK0mlSGrNvy/Wodw0zto6uNG4M49u5AejOtfNounbHJ5K
KYNxC199sbNizoMvp429B6U+5FvwytJWXljJBcpl5LR1/sZC7F9Kbgs8W3F4
PJXYfWuG797QfL14Bmu6o8VTlKKF5+OTkt0AKzGJWZUH8m90PyDv1M2ECwIf
vSEudXkhAae8l5X/FYz00LlbUu3b6ZdOca0SZwlc/VditfY/2nX7Bg==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{380, 125},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]//TableForm=",
 CellID->52991231]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Preprocessors", "Section",
 CellID->12996145],

Cell["\<\
The capabilities of all strategies are extended through symbolic \
preprocessing of the integrals. The preprocessors can be seen as strategies \
that delegate integration to other strategies (preprocessors included).\
\>", "Text",
 CellID->722820349],

Cell[CellGroupData[{

Cell["\"SymbolicPiecewiseSubdivision\"", "Subsection",
 CellID->440163298],

Cell[TextData[{
 Cell[BoxData["\"\<SymbolicPiecewiseSubdivision\>\""], "InlineFormula"],
 " is a preprocessor that divides an integral with a piecewise integrand into \
integrals with disjoint integration regions on each of which the integrand is \
not piecewise. "
}], "Text",
 CellID->99262566],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
integration strategy or preprocessor to which the integration will passed to\
\>", "TableText"]},
   {"\"\<ExpandSpecialPiecewise\>\"", 
    ButtonBox["True",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/True"], Cell[
    "which piecewise functions should be expanded ", "TableText"]},
   {
    ButtonBox["TimeConstraint",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/TimeConstraint"], "5", Cell["\<\
the maximum number of seconds for which the piecewise subdivision will be \
attempted\
\>", "TableText"]},
   {"\"\<MaxPiecewiseCases\>\"", "100", Cell["\<\
the maximum number of subregions the piecewise preprocessor can return\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic preprocessing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->259575088],

Cell[TextData[{
 "Options of ",
 Cell[BoxData["\"\<SymbolicPiecewiseSubdivision\>\""], "InlineFormula"],
 "."
}], "Caption",
 CellID->293336563],

Cell[TextData[{
 "As it was mentioned at the beginning of the tutorial, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " is able to integrate simultaneously integrals with disjoint domains each \
having a different integrand. Hence, after the preprocessing with ",
 Cell[BoxData["\"\<SymbolicPiecewiseSubdivision\>\""], "InlineFormula"],
 " the integration continues in the same way as if, say, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " was given ranges with singularity specifications (which can be seen as \
specifying integrals with disjoint domains with the same integrand). For \
example, the strategy ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " tries to improve the integral estimate of the region with the largest \
error through bisection, and will choose that largest error region regardless \
to what integrand it corresponds to. "
}], "Text",
 CellID->421355523],

Cell[TextData[{
 "Below are the sampling points for the numerical estimation of the integral \
",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], 
    RowBox[{
     RowBox[{"\[Piecewise]", GridBox[{
        {
         FractionBox["2", 
          RowBox[{" ", 
           SqrtBox[
            RowBox[{"sin", "(", 
             RowBox[{"-", "x"}], ")"}]]}]], 
         RowBox[{"x", "<", "0"}]},
        {
         FractionBox["1", 
          SqrtBox["x"]], 
         RowBox[{"x", "\[GreaterEqual]", "0"}]}
       },
       GridBoxAlignment->{
        "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}},
          "RowsIndexed" -> {}},
       GridBoxSpacings->{"Columns" -> {
           Offset[0.27999999999999997`], {
            Offset[0.84]}, 
           Offset[0.27999999999999997`]}, "ColumnsIndexed" -> {}, "Rows" -> {
           Offset[0.2], {
            Offset[0.4]}, 
           Offset[0.2]}, "RowsIndexed" -> {}}]}], 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ". On the plot, the integrand is sampled at the ",
 Cell[BoxData[
  StyleBox["x", "TI"]], "InlineFormula"],
 " coordinates in the order of the ",
 StyleBox["ord", "InlineCode"],
 " coordinates. It can be seen that ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " alternates sampling for the piece ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["2", 
     RowBox[{" ", 
      SqrtBox[
       RowBox[{"sin", "(", 
        RowBox[{"-", "x"}], ")"}]]}]], ",", 
    RowBox[{"x", "<", "0"}]}], TraditionalForm]], "InlineMath"],
 " ",
 "with sampling for the piece ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    FractionBox["1", 
     SqrtBox["x"]], ",", 
    RowBox[{"x", "\[GreaterEqual]", "0", " "}]}], TraditionalForm]], 
  "InlineMath"],
 "."
}], "MathCaption",
 CellID->933325661],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"pnts", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"Piecewise", "[", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            FractionBox["2", 
             SqrtBox[
              RowBox[{"Sin", "[", 
               RowBox[{"-", "x"}], "]"}]]], ",", 
            RowBox[{"x", "<", "0"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            FractionBox["1", 
             SqrtBox["x"]], ",", 
            RowBox[{"x", "\[GreaterEqual]", " ", "0"}]}], "}"}]}], "}"}], 
        "]"}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", 
         RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
       RowBox[{"PrecisionGoal", "\[Rule]", "8"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", "x", "]"}]}]}], "]"}], "]"}], "[", 
    RowBox[{"[", 
     RowBox[{"2", ",", "1"}], "]"}], "]"}]}], ";"}], "\n", 
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Transpose", "[", 
    RowBox[{"{", 
     RowBox[{"pnts", ",", 
      RowBox[{"Range", "[", 
       RowBox[{"Length", "[", "pnts", "]"}], "]"}]}], "}"}], "]"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"AxesOrigin", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "0"}], "}"}]}], ",", 
   RowBox[{"AxesLabel", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"x", ",", "\"\<sampling\\norder\>\""}], "}"}]}]}], 
  "]"}]}], "Input",
 CellLabel->"In[12]:=",
 CellID->53019988],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdlntw1PUVxRdxLGVGGyxC6zAYKApCRgMIhIfmghDekBACIQnJZpNsNo9N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     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesLabel->{
    FormBox["x", TraditionalForm], 
    FormBox["\"sampling\\norder\"", TraditionalForm]},
  AxesOrigin->{-1, 0},
  ImageMargins->0.,
  ImageSize->Automatic,
  PlotRange->{All, All},
  PlotRangeClipping->True,
  PlotRangePadding->{Automatic, Automatic}]], "Output",
 ImageSize->{364, 251},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->233472080]
}, Open  ]],

Cell[TextData[{
 "Here are the sampling points for the numerical estimation of the integral \
",
 Cell[BoxData[
  RowBox[{
   SubsuperscriptBox["\[Integral]", 
    RowBox[{"-", 
     StyleBox["\[Pi]", "TR"]}], 
    StyleBox["\[Pi]", "TR"]], 
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", 
      StyleBox["\[Pi]", "TR"]}], 
     StyleBox["\[Pi]", "TR"]], 
    RowBox[{
     RowBox[{
      ButtonBox["Boole",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Boole"], "[", 
      RowBox[{
       RowBox[{
        SuperscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["2", "TR"]], "+", 
        SuperscriptBox[
         StyleBox["y", "TI"], 
         StyleBox["2", "TR"]]}], ">", 
       StyleBox["1", "TR"]}], "]"}], " ", 
     StyleBox["sin", "TR"], 
     RowBox[{"(", 
      RowBox[{
       SuperscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["2", "TR"]], "+", 
       SuperscriptBox[
        StyleBox["y", "TI"], 
        StyleBox["2", "TR"]]}], ")"}], 
     RowBox[{
      StyleBox["\[DifferentialD]", "TR"], 
      StyleBox["y", "TI"]}], 
     RowBox[{
      StyleBox["\[DifferentialD]", "TR"], 
      StyleBox["x", "TI"]}]}]}]}]], "InlineFormula"],
 ". The integrand is plotted on the left, the sampling points are plotted on \
the right. The integral has been partitioned into ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "\[Pi]"}], 
     RowBox[{"-", "1"}]], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"-", "\[Pi]"}], "\[Pi]"], 
     RowBox[{
      RowBox[{"sin", "(", 
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ")"}], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " + ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"-", "\[Pi]"}], 
      RowBox[{"-", 
       SqrtBox[
        RowBox[{"1", "-", 
         SuperscriptBox["x", "2"]}]]}]], 
     RowBox[{
      RowBox[{"sin", "(", 
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ")"}], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " + ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      SqrtBox[
       RowBox[{"1", "-", 
        SuperscriptBox["x", "2"]}]], "\[Pi]"], 
     RowBox[{
      RowBox[{"sin", "(", 
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ")"}], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}], " "}]}]}], TraditionalForm]], 
  "InlineMath"],
 " + ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "1", "\[Pi]"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"-", "\[Pi]"}], "\[Pi]"], 
     RowBox[{
      RowBox[{"sin", "(", 
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], ")"}], 
      RowBox[{"\[DifferentialD]", "y"}], 
      RowBox[{"\[DifferentialD]", "x"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", that is why the sampling points form a different pattern for ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"-", "1"}], "\[LessEqual]", " ", "x", "\[LessEqual]", "1"}], 
   TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->722260207],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"Boole", "[", 
       RowBox[{
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"]}], ">", "1"}], "]"}], " ", 
      RowBox[{"Sin", "[", 
       RowBox[{
        SuperscriptBox["x", "2"], "+", 
        SuperscriptBox["y", "2"]}], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", " ", "\[Pi]"}], ",", " ", "\[Pi]"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "\[Pi]"}], ",", " ", "\[Pi]"}], "}"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"grSP", "=", 
   RowBox[{"Point", "/@", 
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"Boole", "[", 
          RowBox[{
           RowBox[{
            SuperscriptBox["x", "2"], "+", 
            SuperscriptBox["y", "2"]}], ">", "1"}], "]"}], " ", 
         RowBox[{"Sin", "[", 
          RowBox[{
           SuperscriptBox["x", "2"], "+", 
           SuperscriptBox["y", "2"]}], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", " ", "\[Pi]"}], ",", " ", "\[Pi]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<SymbolicPiecewiseSubdivision\>\"", ",", 
           RowBox[{"Method", "\[Rule]", "\"\<GlobalAdaptive\>\""}]}], "}"}]}],
         ",", 
        RowBox[{"PrecisionGoal", "\[Rule]", "3"}], ",", 
        RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
         RowBox[{"Sow", "[", 
          RowBox[{"{", 
           RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
     RowBox[{"[", 
      RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"grSP", "=", 
   RowBox[{"Graphics", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.005", "]"}], ",", "grSP"}], "}"}], ",", 
     RowBox[{"Axes", "\[Rule]", "True"}], ",", 
     RowBox[{"AxesOrigin", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", " ", "\[Pi]"}], ",", 
        RowBox[{"-", " ", "\[Pi]"}]}], "}"}]}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"GraphicsArray", "[", 
  RowBox[{"{", 
   RowBox[{"gr", ",", "grSP"}], "}"}], "]"}]}], "Input",
 CellLabel->"In[14]:=",
 CellID->577543440],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 90},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[17]=",
 CellID->5885665]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"ExpandSpecialPiecewise\"", "Subsubsection",
 CellID->32368822],

Cell[TextData[{
 "In some cases it is preferable to do piecewise expansion only over certain \
piecewise functions. In these case the option ",
 Cell[BoxData["\"\<ExpandSpecialPiecewise\>\""], "InlineFormula"],
 " can be given a list of functions to do the piecewise expansion with."
}], "Text",
 CellID->222221023],

Cell[TextData[{
 "This Monte Carlo integral is done faster with piecewise expansion only over \
",
 Cell[BoxData[
  ButtonBox["Boole",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Boole"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->8352495],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x_", ",", "y_"}], "]"}], ":=", 
   RowBox[{
    RowBox[{"Boole", "[", 
     RowBox[{
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       RowBox[{"2", 
        SuperscriptBox["y", "2"]}]}], "<", "1"}], "]"}], 
    RowBox[{"Abs", "[", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "3"], "-", "2"}], "]"}], 
    RowBox[{"Abs", "[", 
     RowBox[{
      RowBox[{"-", 
       SuperscriptBox["x", "2"]}], "+", 
      SuperscriptBox["y", "2"], "+", "1"}], "]"}], 
    RowBox[{"Abs", "[", 
     RowBox[{
      SuperscriptBox["x", "2"], "-", 
      RowBox[{"3", 
       SuperscriptBox["y", "2"]}], "+", "x"}], "]"}], 
    FractionBox["1", 
     SqrtBox[
      RowBox[{
       SuperscriptBox["x", "2"], "+", 
       SuperscriptBox["y", "2"], "+", "10"}]]]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"x", ",", "y"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<SymbolicPiecewiseSubdivision\>\"", ",", 
       RowBox[{"\"\<ExpandSpecialPiecewise\>\"", "\[Rule]", 
        RowBox[{"{", "Boole", "}"}]}], ",", 
       RowBox[{"Method", "\[Rule]", "\"\<MonteCarlo\>\""}]}], "}"}]}]}], 
   "]"}], "//", "Timing"}]}], "Input",
 CellLabel->"In[18]:=",
 CellID->73439856],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.10898399999999946`", ",", "0.6347213810258568`"}], 
  "}"}]], "Output",
 ImageSize->{142, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[19]=",
 CellID->749686954]
}, Open  ]],

Cell[TextData[{
 "Here is a Monte Carlo integration with piecewise expansion over both ",
 Cell[BoxData[
  ButtonBox["Boole",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Boole"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Abs",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Abs"]], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->161065457],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    RowBox[{"f", "[", 
     RowBox[{"x", ",", "y"}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"y", ",", 
      RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<SymbolicPiecewiseSubdivision\>\"", ",", 
       RowBox[{"Method", "\[Rule]", "\"\<MonteCarlo\>\""}]}], "}"}]}]}], 
   "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[20]:=",
 CellID->817526177],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.1919700000000007`", ",", "0.6251636639947449`"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[20]=",
 CellID->35996428]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"EvenOddSubdivision\"", "Subsection",
 CellID->172715735],

Cell[TextData[{
 Cell[BoxData["\"\<EvenOddSubdivision\>\""], "InlineFormula"],
 " is a preprocessor that reduces the integration region if the region is \
symmetric around the origin and the integrand is determined to be even or \
odd. The convergence of odd integrals is verified by default."
}], "Text",
 CellID->110859602],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
integration strategy or preprocessor to which the integration will passed to\
\>", "TableText"]},
   {"VerifyConvergence", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
should the convergence be verified if odd integral is detected\
\>", "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic processing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->87955492],

Cell[TextData[{
 "Options of ",
 Cell[BoxData["\"\<EvenOddSubdivision\>\""], "InlineFormula"],
 "."
}], "Caption",
 CellID->225358991],

Cell["\<\
When the integrand is an even function and the integration region is \
symmetric around the origin, the integral can be computed by integrating only \
on some part of the integration region and multiplying with a corresponding \
factor.\
\>", "Text",
 CellID->445675680],

Cell[TextData[{
 "Here is a plot of an even function and ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s sampling points without any preprocessing."
}], "MathCaption",
 CellID->110699463],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"Plot3D", "[", 
    RowBox[{
     RowBox[{"1", "+", 
      RowBox[{"Sin", "[", 
       SqrtBox[
        RowBox[{
         SuperscriptBox["x", "2"], "+", 
         SuperscriptBox["y", "2"]}]], "]"}]}], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", 
       RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"y", ",", 
       RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}]}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"grSP", "=", 
   RowBox[{"Point", "[", 
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{
        RowBox[{"1", "+", 
         RowBox[{"Sin", "[", 
          SqrtBox[
           RowBox[{
            SuperscriptBox["x", "2"], "+", 
            SuperscriptBox["y", "2"]}]], "]"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", 
          RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"Automatic", ",", 
           RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}], 
        ",", 
        RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
         RowBox[{"Sow", "[", 
          RowBox[{"{", 
           RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], 
     "\[LeftDoubleBracket]", 
     RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}], "]"}]}], 
  ";"}], "\n", 
 RowBox[{
  RowBox[{"grSP", "=", 
   RowBox[{"Graphics", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"PointSize", "[", "0.01", "]"}], ",", "grSP"}], "}"}], ",", 
     RowBox[{"Axes", "\[Rule]", "True"}], ",", 
     RowBox[{"AxesOrigin", "\[Rule]", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "\[Pi]"}], ",", 
        RowBox[{"-", "\[Pi]"}]}], "}"}]}]}], "]"}]}], ";"}], "\n", 
 RowBox[{"GraphicsGrid", "[", 
  RowBox[{"{", 
   RowBox[{"{", 
    RowBox[{"gr", ",", "grSP"}], "}"}], "}"}], "]"}]}], "Input",
 CellLabel->"In[21]:=",
 CellID->484464303],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 90},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[24]=",
 CellID->128779153]
}, Open  ]],

Cell[TextData[{
 "These are the sampling points used by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " after ",
 Cell[BoxData["\"\<EvenOddSubdivision\>\""], "InlineFormula"],
 " has been applied. Note that the sampling points are only in the region ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"[", 
     RowBox[{"0", ",", "\[Pi]"}], "]"}], "\[Times]", 
    RowBox[{"[", 
     RowBox[{"0", ",", "\[Pi]"}], "]"}]}], TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->160780120],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"PointSize", "[", "0.01", "]"}], ",", 
     RowBox[{"Point", "/@", 
      RowBox[{
       RowBox[{"Reap", "[", 
        RowBox[{"NIntegrate", "[", 
         RowBox[{
          RowBox[{"1", "+", 
           RowBox[{"Sin", "[", 
            SqrtBox[
             RowBox[{
              SuperscriptBox["x", "2"], "+", 
              SuperscriptBox["y", "2"]}]], "]"}]}], ",", 
          RowBox[{"{", 
           RowBox[{"x", ",", 
            RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"y", ",", 
            RowBox[{"-", "\[Pi]"}], ",", "\[Pi]"}], "}"}], ",", 
          RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
           RowBox[{"Sow", "[", 
            RowBox[{"{", 
             RowBox[{"x", ",", "y"}], "}"}], "]"}]}]}], "]"}], "]"}], "[", 
       RowBox[{"[", 
        RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}], "}"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[25]:=",
 CellID->151551521],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzlWs1u2zAMViw7P0vaFVn3c9mSYj3vsIfYTnuIoBjQYig2dH2FYbcO2BOs
T+pZiWXLn2hRtmUn2Qo0MSVSokjxIyXn0+b++vPt5v7marP+eLf5dn1z9X39
4etd1iRHQojH7P/PK5E9p9nT9v8xTbNv+/Mi+4wyBqkoqThVqxCqbZE9q6bI
6FOPqu+E6NNy8+w5rpFbEH2mnARdtNx4y7dWn3HZI8QEeJ8ALS2JE+AYAz0D
zWJrhAVwoA5z1N8aIRlsNd1HOKTV8LqcAkcCtPJupHX2WkWSS6BuvPZxLok6
8VpL7zldWpuSC3YkSnvLexUJOzLmrB44S91u0rg0yb/n+ffYsEyTfqW9NPp3
eGIi4Rlo9uBoVf76nTWOCp6Iwjv1kRT0bnmFby2U42htcWtE9BTDmRTIRbS2
54304sTuL9yosrByIuzMMfWyiY7iqWe7mkXCihDPzDkEMYYSjWtXN2akI2v1
UbGw7V9av+uo/YXoOwN6U8xj++cN8K5IrjVwvYYVxtWVJOZKBLGSH6VGsLu1
bGlNrF/Q/zZW4WyIVM+BRv+8YPpRfkyuBvX8WbER6lTt5S1oxqSm6d3Ioc8X
UvsSI/XI1M5DKbSU5WGGP/T4hmeceAWtjjrlJdAYGRdALz2s2yReEIvtLFvn
K0SJWw/N3LvUqvSRFmVOjE0ZUl+yhnTQU5gNUQ/50bMTkO8yv3qsVgyYUVxZ
yo5G08PU+Q2lHnJTmnZFDWzLkzU3u2M8ZnFYCem3QD9j+lF+7qmTIp9CO9bm
2I+xjf1ILwldtN/RY4grHN0Od9wVO0ZEvougLioyC1tB2yePugqma6ZSY/wi
dTA0szVxWgNnO9S6oorQhhUCxLZ7riaYptpxNDPeBBOPgpFXjyXiSkIL6+yU
lnnG/EY+xHkdA2ZccHWLznqzXLemc1VPBpQdhrUvH/tcK7V6Ll6GjCfj9EUh
EYEXOznktDGQxs5+5RQDlyGw/7KlXlzW9MtU3eftKZsbUa9vmOid4J2XGkYN
1rUchkzTf7GeDm1Vejzu5g7XjDbCNXC+a3oe5s50XefnKo4wVuYQd0sTcYar
30/+IG3msCmN25SUIptxUxhIVQb6ztxEM8TL03zMOllJaqZP766ab7hahYtX
+o56CVwYNVtkq+Sic3Kcc9AZx8FoPQP+PCBqzl9d/E1ZrFu1jdl0BTT1dsxv
h/VXb+LuCH17Gfp810/98I6wvgtlA97BBqzemt6ztstRdb91qLt/Sw04sO8P
dBQPdVKlvLWfKq7taYesx0ibNuGnPGfx+Eh5nA1w374H2u8U4roJGCpquLGw
/7LB2CFq8+NEc5+zzWFlftQE7ah/h8PdQFI16XD3amHygdLZNx9wb0qOY383
O8fgmrg7d1pKe93nrVcf7ziGQ95w2cLl2X7ewXXHuP+vLuojWzQdj7sj4nY6
8nPniX3eYUlB1Ii1yNK2mmxdhRqoZP+KVVe6x5knwlb8/f6OTTXwd/2SGG+F
2osutyCHU/d1Qxox+gtgJVez\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{161, 160},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[25]=",
 CellID->104119152]
}, {2}]],

Cell[CellGroupData[{

Cell["Transformation theorem", "Subsubsection",
 CellID->284148603],

Cell[TextData[{
 "The preprocessor ",
 Cell[BoxData["\"\<EvenOddSubdivision\>\""], "InlineFormula"],
 " is based on the following theorem."
}], "Text",
 CellID->339656564],

Cell[TextData[{
 StyleBox["Theorem",
  FontWeight->"Bold"],
 ": Given the ",
 Cell[BoxData[
  FormBox["d", TraditionalForm]], "InlineMath"],
 "-dimensional integral"
}], "Text",
 CellID->160419956],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     SubscriptBox["a", "0"], 
     SubscriptBox["b", "0"]], 
    RowBox[{"\[Ellipsis]", 
     RowBox[{
      SubsuperscriptBox["\[Integral]", 
       RowBox[{
        SubscriptBox["a", "i"], "(", 
        RowBox[{
         SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "1"}]]}], ")"}], 
       RowBox[{
        SubscriptBox["b", "i"], "(", 
        RowBox[{
         SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
         SubscriptBox["x", 
          RowBox[{"i", "-", "1"}]]}], ")"}]], 
      RowBox[{"\[Ellipsis]", 
       RowBox[{
        SubsuperscriptBox["\[Integral]", 
         RowBox[{
          SubscriptBox["a", "n"], "(", 
          RowBox[{
           SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
           SubscriptBox["x", 
            RowBox[{"d", "-", "1"}]]}], ")"}], 
         RowBox[{
          SubscriptBox["b", "n"], "(", 
          RowBox[{
           SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
           SubscriptBox["x", 
            RowBox[{"d", "-", "1"}]]}], ")"}]], 
        RowBox[{
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
           SubscriptBox["x", "d"]}], ")"}], 
         RowBox[{"\[DifferentialD]", 
          SubscriptBox["x", "1"]}], "\[Ellipsis]", 
         RowBox[{"\[DifferentialD]", 
          SubscriptBox["x", "d"]}]}]}]}]}]}]}], ","}], 
  TraditionalForm]], "DisplayMath",
 CellID->29357659],

Cell[TextData[{
 "assume that for some ",
 Cell[BoxData[
  FormBox[
   RowBox[{"i", "\[Element]", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "\[Ellipsis]", ",", "d"}], "}"}]}], 
   TraditionalForm]], "InlineMath"],
 " these equalities hold:"
}], "Text",
 CellID->82597417],

Cell[TextData[{
 "a) ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     SubscriptBox["a", "i"], "(", 
     RowBox[{
      SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", 
       RowBox[{"i", "-", "1"}]]}], ")"}], "=", 
    RowBox[{"-", 
     RowBox[{
      SubscriptBox["b", "i"], "(", 
      RowBox[{
       SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
       SubscriptBox["x", 
        RowBox[{"i", "-", "1"}]]}], ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 ","
}], "Text",
 CellID->311627064],

Cell[TextData[{
 "b) for all ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"j", ">", "i"}], ",", " ", 
    FormBox[
     RowBox[{"j", "\[Element]", 
      RowBox[{"{", 
       RowBox[{"1", ",", "2", ",", "\[Ellipsis]", ",", "d"}], "}"}]}],
     TraditionalForm]}], TraditionalForm]], "InlineMath"],
 ":"
}], "Text",
 CellID->107898797],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{
     SubscriptBox["a", "j"], "(", 
     RowBox[{
      SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", "i"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", 
       RowBox[{"j", "-", "1"}]]}], ")"}], "=", 
    RowBox[{
     SubscriptBox["a", "j"], "(", 
     RowBox[{
      SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
      RowBox[{"-", 
       SubscriptBox["x", "i"]}], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", 
       RowBox[{"j", "-", "1"}]]}], ")"}]}], ",", " ", "\[IndentingNewLine]", 
   RowBox[{
    RowBox[{
     SubscriptBox["b", "j"], "(", 
     RowBox[{
      SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", "i"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", 
       RowBox[{"j", "-", "1"}]]}], ")"}], "=", 
    RowBox[{
     RowBox[{
      SubscriptBox["b", "j"], "(", 
      RowBox[{
       SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
       RowBox[{"-", 
        SubscriptBox["x", "i"]}], ",", "\[Ellipsis]", ",", 
       SubscriptBox["x", 
        RowBox[{"j", "-", "1"}]]}], ")"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->750599169],

Cell[TextData[{
 "In other words the range of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "i"], " "}], TraditionalForm]], "InlineMath"],
 " is symmetric around the origin, and the boundaries of the variables ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["x", "j"], ",", " ", 
    RowBox[{"j", ">", "i"}]}], TraditionalForm]], "InlineMath"],
 " are even functions wrt ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->96819779],

Cell["Then:", "Text",
 CellID->429793367],

Cell["a) the integral is equivalent to", "Text",
 CellID->182878094],

Cell[BoxData[
 FormBox[
  RowBox[{"2", 
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     SubscriptBox["a", "0"], 
     SubscriptBox["b", "0"]], 
    RowBox[{"\[Ellipsis]", 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "0", 
       RowBox[{
        SubscriptBox["b", "i"], "(", 
        RowBox[{
         SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
         SubscriptBox["x", "i"]}], ")"}]], 
      RowBox[{"\[Ellipsis]", 
       RowBox[{
        SubsuperscriptBox["\[Integral]", 
         RowBox[{
          SubscriptBox["a", "d"], "(", 
          RowBox[{
           SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
           SubscriptBox["x", 
            RowBox[{"d", "-", "1"}]]}], ")"}], 
         RowBox[{
          SubscriptBox["b", "d"], "(", 
          RowBox[{
           SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
           SubscriptBox["x", 
            RowBox[{"d", "-", "1"}]]}], ")"}]], 
        RowBox[{
         RowBox[{"f", "(", 
          RowBox[{
           SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
           SubscriptBox["x", "d"]}], ")"}], 
         RowBox[{"\[DifferentialD]", 
          SubscriptBox["x", "1"]}], "\[Ellipsis]", 
         RowBox[{"\[DifferentialD]", 
          SubscriptBox["x", "d"]}]}]}]}]}]}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->212459658],

Cell[TextData[{
 "if the integrand is even wrt ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 ", that is,"
}], "Text",
 CellID->531012682],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{
    RowBox[{"f", "(", 
     RowBox[{
      SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", "i"], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", "d"]}], ")"}], "=", 
    RowBox[{"f", "(", 
     RowBox[{
      SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
      RowBox[{"-", 
       SubscriptBox["x", "i"]}], ",", "\[Ellipsis]", ",", 
      SubscriptBox["x", "d"]}], ")"}]}], ";"}], 
  TraditionalForm]], "DisplayMath",
 CellID->25665149],

Cell[TextData[{
 "b) the integral is equivalent to ",
 Cell[BoxData[
  FormBox["0", TraditionalForm]], "InlineMath"],
 ", if the integrand is odd wrt ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 ", that is, "
}], "Text",
 CellID->3961441],

Cell[BoxData[
 FormBox[
  RowBox[{
   RowBox[{"f", "(", 
    RowBox[{
     SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["x", "i"], ",", "\[Ellipsis]", ",", 
     SubscriptBox["x", "d"]}], ")"}], "=", 
   RowBox[{"-", 
    RowBox[{
     RowBox[{"f", "(", 
      RowBox[{
       SubscriptBox["x", "1"], ",", "\[Ellipsis]", ",", 
       RowBox[{"-", 
        SubscriptBox["x", "i"]}], ",", "\[Ellipsis]", ",", 
       SubscriptBox["x", "d"]}], ")"}], "."}]}]}], 
  TraditionalForm]], "DisplayMath",
 CellID->108641785],

Cell["\<\
Note that the theorem above can be applied several times over an integral. \
\>", "Text",
 CellID->127597917],

Cell[TextData[{
 "To illustrate the theorem consider the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"-", "x"}], "x"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "2", 
       SuperscriptBox["y", "2"]], 
      RowBox[{"x", 
       RowBox[{"\[DifferentialD]", "z"}], 
       RowBox[{"\[DifferentialD]", "y"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ". It is symmetric along ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 ", and the integrand and the bounds of ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " are even functions wrt to ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->19683496],

Cell[TextData[{
 "Here is a plot of the sampling points without the application of ",
 Cell[BoxData["\"\<EvenOddSubdivision\>\""], "InlineFormula"],
 " (black) and with ",
 Cell[BoxData["\"\<EvenOddSubdivision\>\""], "InlineFormula"],
 " applied (red)."
}], "MathCaption",
 CellID->145120706],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"grEven", "=", 
   RowBox[{"Point", "/@", 
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{"x", ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "x"}], ",", "x"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"z", ",", "2", ",", 
          SuperscriptBox["y", "2"]}], "}"}], ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<SymbolicPreprocessing\>\"", ",", 
           RowBox[{"\"\<UnitCubeRescaling\>\"", "\[Rule]", "False"}], ",", 
           RowBox[{"Method", "->", 
            RowBox[{"{", 
             RowBox[{"\"\<LobattoKronrodRule\>\"", ",", 
              RowBox[{"\"\<GaussPoints\>\"", "\[Rule]", "5"}]}], "}"}]}]}], 
          "}"}]}], ",", 
        RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
         RowBox[{"Sow", "[", 
          RowBox[{"{", 
           RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]}]}], "]"}], "]"}], 
     "[", 
     RowBox[{"[", 
      RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"gr", "=", 
   RowBox[{"Point", "/@", 
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{"x", ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "x"}], ",", "x"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"z", ",", "2", ",", 
          SuperscriptBox["y", "2"]}], "}"}], ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<LobattoKronrodRule\>\"", ",", 
           RowBox[{"\"\<GaussPoints\>\"", "\[Rule]", "5"}], ",", 
           RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}], 
        ",", 
        RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
         RowBox[{"Sow", "[", 
          RowBox[{"{", 
           RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]}]}], "]"}], "]"}], 
     "[", 
     RowBox[{"[", 
      RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"gr", ",", "Red", ",", "grEven"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}], ",", 
   RowBox[{"ViewPoint", "->", 
    RowBox[{"{", 
     RowBox[{"2.813", ",", " ", "0.765", ",", " ", "1.718"}], "}"}]}]}], 
  "]"}]}], "Input",
 CellLabel->"In[26]:=",
 CellID->447116046],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{286, 300},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[28]=",
 CellID->378907499]
}, Open  ]],

Cell[TextData[{
 "If the bounds of ",
 Cell[BoxData[
  FormBox["z", TraditionalForm]], "InlineMath"],
 " were not even functions wrt to ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " then the symmetry along ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " is broken. For example, the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "1"], 
    RowBox[{
     SubsuperscriptBox["\[Integral]", 
      RowBox[{"-", "x"}], "x"], 
     RowBox[{
      SubsuperscriptBox["\[Integral]", "2", "y"], 
      RowBox[{"x", 
       RowBox[{"\[DifferentialD]", "z"}], 
       RowBox[{"\[DifferentialD]", "y"}], 
       RowBox[{"\[DifferentialD]", "x"}]}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " has no symmetry ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can exploit. "
}], "Text",
 CellID->6925575],

Cell[TextData[{
 "Here is a plot of the sampling points with ",
 Cell[BoxData["\"\<EvenOddSubdivision\>\""], "InlineFormula"],
 " applied (red). The region has no symmetry along ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 "."
}], "MathCaption",
 CellID->152458070],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"grEven", "=", 
   RowBox[{"Point", "/@", 
    RowBox[{
     RowBox[{"Reap", "[", 
      RowBox[{"NIntegrate", "[", 
       RowBox[{"x", ",", 
        RowBox[{"{", 
         RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"y", ",", 
          RowBox[{"-", "x"}], ",", "x"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"z", ",", "2", ",", "y"}], "}"}], ",", 
        RowBox[{"Method", "\[Rule]", 
         RowBox[{"{", 
          RowBox[{"\"\<SymbolicPreprocessing\>\"", ",", 
           RowBox[{"\"\<UnitCubeRescaling\>\"", "\[Rule]", "False"}], ",", 
           RowBox[{"Method", "->", 
            RowBox[{"{", 
             RowBox[{"\"\<LobattoKronrodRule\>\"", ",", 
              RowBox[{"\"\<GaussPoints\>\"", "\[Rule]", "5"}]}], "}"}]}]}], 
          "}"}]}], ",", 
        RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
         RowBox[{"Sow", "[", 
          RowBox[{"{", 
           RowBox[{"x", ",", "y", ",", "z"}], "}"}], "]"}]}]}], "]"}], "]"}], 
     "[", 
     RowBox[{"[", 
      RowBox[{"2", ",", "1"}], "]"}], "]"}]}]}], ";"}], "\n", 
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"Red", ",", "grEven"}], "}"}], ",", 
   RowBox[{"PlotRange", "\[Rule]", "All"}], ",", 
   RowBox[{"Axes", "\[Rule]", "True"}], ",", 
   RowBox[{"ViewPoint", "->", 
    RowBox[{"{", 
     RowBox[{"2.813", ",", " ", "0.765", ",", " ", "1.718"}], "}"}]}]}], 
  "]"}]}], "Input",
 CellLabel->"In[29]:=",
 CellID->36445814],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{203, 290},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[30]=",
 CellID->207663192]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"VerifyConvergence\"", "Subsubsection",
 CellID->302014659],

Cell[TextData[{
 "Consider the following divergent integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "\[Infinity]"}], "\[Infinity]"], 
    RowBox[{"x", 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 ". ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " detects it as an odd function over a symmetric domain and tries to \
integrate ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "\[Infinity]"], 
    RowBox[{"x", 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 " (that is, check the convergence of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "\[Infinity]"], 
    RowBox[{"x", 
     RowBox[{"\[DifferentialD]", "x"}]}]}], TraditionalForm]], "InlineMath"],
 "). Since no convergence was reached as it is indicated by the ",
 StyleBox["ncvb", "InlineCode"],
 " message, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " gives the message ",
 StyleBox["oidiv", "InlineCode"],
 " that the integral might be divergent."
}], "MathCaption",
 CellID->242556561],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{"x", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[31]:=",
 CellID->475092308],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[31]=",
 CellID->111979644]
}, Open  ]],

Cell[TextData[{
 "If the option ",
 Cell[BoxData["VerifyConvergence"], "InlineFormula"],
 " is set to ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " no convergence verification\[LongDash]and hence no integrand evaluation",
 "\[LongDash]",
 "will be done after the integral is found to be odd."
}], "MathCaption",
 CellID->97926081],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{"x", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<EvenOddSubdivision\>\"", ",", 
      RowBox[{"\"\<VerifyConvergence\>\"", "\[Rule]", "False"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[32]:=",
 CellID->17795442],

Cell[BoxData["0.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[32]=",
 CellID->684364607]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\"OscillatorySelection\"", "Subsection",
 CellID->52803663],

Cell[TextData[{
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " is a preprocessor that selects specialized algorithms for efficient \
evaluation of ",
 "one-dimensional",
 " oscillating integrals, the integrands of which are products of a \
trigonometric or Bessel function and a non-oscillating or a (much) slower \
oscillating function."
}], "Text",
 CellID->223870075],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {"\"\<BesselInfiniteRangeMethod\>\"", "\"\<ExtrapolatingOscillatory\>\"", 
    "\[SpanFromLeft]"},
   {"", Cell[" ", "TableText"], Cell["\<\
specialized integration algorithm for infinite region integrals with Bessel \
functions\
\>", "TableText"]},
   {"\"\<FourierFiniteRangeMethod\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell["\<\
specialized integration algorithm for Fourier integrals over finite ranges\
\>", "TableText"]},
   {"\"\<FourierInfiniteRangeMethod\>\"", 
    RowBox[{"{", 
     RowBox[{"\"\<DoubleExponentialOscillatory\>\"", ",", " ", 
      RowBox[{
       ButtonBox["Method",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/Method"], "->", 
       "\"\<ExtrapolatingOscillatory\>\""}]}], "}"}], "\[SpanFromLeft]"},
   {"", " ", Cell["\<\
specialized integration algorithm for Fourier integrals over infinite regions\
\
\>", "TableText"]},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], "\"\<GlobalAdaptive\>\"", Cell["\<\
integration strategy or preprocessor to which the integration will passed to\
\>", "TableText"]},
   {"\"\<TermwiseOscillatory\>\"", 
    ButtonBox["False",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/False"], Cell[TextData[{
     "if the value of this option is ",
     Cell[BoxData[
      ButtonBox["True",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/True"]], "InlineFormula"],
     " then the algorithm is selected for each term\nin a sum of oscillatory \
functions "
    }], "TableText"]},
   {"\"\<SymbolicProcessing\>\"", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "number of seconds to do symbolic processing", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->138978972],

Cell[TextData[{
 "Options of ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 "."
}], "Caption",
 CellID->79938395],

Cell[TextData[{
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " is used by default."
}], "MathCaption",
 CellID->14738821],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox[
    RowBox[{"Sin", "[", 
     RowBox[{"20", "x"}], "]"}], 
    RowBox[{"x", "+", "1"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[33]:=",
 CellID->16892581],

Cell[BoxData["0.049757004074143144`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[33]=",
 CellID->67834372]
}, Open  ]],

Cell[TextData[{
 "Without the ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " preprocessor ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " does not reach convergence with its default option settings."
}], "MathCaption",
 CellID->87120345],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   FractionBox[
    RowBox[{"Sin", "[", 
     RowBox[{"20", "x"}], "]"}], 
    RowBox[{"x", "+", "1"}]], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"Automatic", ",", 
      RowBox[{"\"\<SymbolicProcessing\>\"", "\[Rule]", "0"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[34]:=",
 CellID->6164165],

Cell[BoxData["0``-2.4270499283213987"], "Output",
 ImageSize->{49, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[34]=",
 CellID->158780197]
}, Open  ]],

Cell[TextData[{
 "The preprocessor ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " is designed to work with (the internal output of) the",
 " ",
 Cell[BoxData["\"\<SymbolicPiecewiseSubdivision\>\""], "InlineFormula"],
 " preprocessor. ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " itself partitions oscillatory integrals that include the origin or have \
oscillatory kernels that need to be expanded or transformed into forms the \
oscillatory algorithms are designed for. "
}], "Text",
 CellID->344602079],

Cell[TextData[{
 "Here is a piecewise function integration in which all methods of ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " are used. For this integral the preprocessor ",
 Cell[BoxData["\"\<SymbolicPiecewiseSubdivision\>\""], "InlineFormula"],
 " divides the integral into four different integrals; for each of these \
integrals ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " selects an appropriate algorithm."
}], "MathCaption",
 CellID->601967181],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"\[Piecewise]", GridBox[{
      {
       FractionBox[
        RowBox[{"BesselJ", "[", 
         RowBox[{"3", ",", 
          RowBox[{"-", "x"}]}], "]"}], 
        SqrtBox[
         RowBox[{"-", "x"}]]], 
       RowBox[{"x", "<", "0"}]},
      {
       FractionBox[
        RowBox[{"Cos", "[", 
         RowBox[{"200", " ", "x"}], "]"}], 
        SqrtBox["x"]], 
       RowBox[{"0", "<", "x", "<", "20"}]},
      {
       FractionBox[
        RowBox[{"Sin", "[", 
         RowBox[{
          RowBox[{"2", " ", "x"}], "+", "3"}], "]"}], 
        SuperscriptBox["x", "2"]], 
       RowBox[{"x", ">", "30"}]},
      {
       FractionBox["1", 
        RowBox[{"Log", "[", "x", "]"}]], 
       TagBox["True",
        "PiecewiseDefault",
        AutoDelete->False,
        DeletionWarning->True]}
     },
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.28], {
          Offset[0.84]}, 
         Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->334870329],

Cell[BoxData["3.779327349359721`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->44547799]
}, Open  ]],

Cell[TextData[{
 "The following table shows the names of the ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " options used to specify the algorithms for each sub-interval in the \
integral above."
}], "MathCaption",
 CellID->30950777],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     RowBox[{
      StyleBox["x", "TI"], "\[Element]", 
      RowBox[{"(", 
       RowBox[{
        RowBox[{"-", "\[Infinity]"}], ",", "0"}]}]}], "]"}], 
    StyleBox["\"\<BesselInfiniteRangeMethod\>\"", "InlineCode"]},
   {
    RowBox[{
     StyleBox["x", "TI"], "\[Element]", 
     RowBox[{"[", 
      RowBox[{"0", ",", "20"}], "]"}]}], 
    StyleBox["\"\<FourierFiniteRangeMethod\>\"", "InlineCode"]},
   {
    RowBox[{
     RowBox[{
      StyleBox["x", "TI"], "\[Element]", 
      RowBox[{"[", 
       RowBox[{"30", ",", "\[Infinity]"}]}]}], ")"}], 
    StyleBox["\"\<FourierInfiniteRangeMethod\>\"", "InlineCode"]},
   {
    RowBox[{
     StyleBox["x", "TI"], "\[Element]", 
     RowBox[{"[", 
      RowBox[{"20", ",", "30"}], "]"}]}], 
    StyleBox["Method", "InlineCode"]}
  }]], "DefinitionBox",
 CellID->304672727],

Cell[TextData[{
 "In this example ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " is called twice",
 ". ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 " is a special algorithm for Fourier integrals, and the formula ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"2", "\[ImaginaryI]", " ", 
      SuperscriptBox["x", "2"]}]], "=", 
    RowBox[{
     RowBox[{"cos", "(", 
      RowBox[{"2", 
       SuperscriptBox["x", "2"]}], ")"}], "+", 
     RowBox[{"\[ImaginaryI]", " ", 
      RowBox[{"sin", "(", 
       RowBox[{"2", 
        SuperscriptBox["x", "2"]}], ")"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 " makes the integrand a sum of two Fourier integrands."
}], "MathCaption",
 CellID->308324465],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Exp", "[", 
      RowBox[{"2", "\[ImaginaryI]", " ", 
       SuperscriptBox["x", "2"]}], "]"}], 
     RowBox[{"x", "+", "1"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[35]:=",
 CellID->179765096],

Cell["0.39934219109623426 + 0.2791805912092563*I", "Output",
 ImageSize->{294, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[35]//InputForm=",
 CellID->132409675]
}, Open  ]],

Cell[TextData[{
 "To demonstrate that ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " has used the formula ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"2", "\[ImaginaryI]", " ", 
      SuperscriptBox["x", "2"]}]], "=", 
    RowBox[{
     RowBox[{"cos", "(", 
      RowBox[{"2", 
       SuperscriptBox["x", "2"]}], ")"}], "+", 
     RowBox[{"\[ImaginaryI]", " ", 
      RowBox[{"sin", "(", 
       RowBox[{"2", 
        SuperscriptBox["x", "2"]}], ")"}]}]}]}], TraditionalForm]], 
  "InlineMath"],
 ", here is the integral above split \"by hand\". The result is identical \
with the last result."
}], "MathCaption",
 CellID->713818563],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"NIntegrate", "[", 
    RowBox[{
     FractionBox[
      RowBox[{"Cos", "[", 
       RowBox[{"2", " ", 
        SuperscriptBox["x", "2"]}], "]"}], 
      RowBox[{"x", "+", "1"}]], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}], "+", 
   RowBox[{"\[ImaginaryI]", " ", 
    RowBox[{"NIntegrate", "[", 
     RowBox[{
      FractionBox[
       RowBox[{"Sin", "[", 
        RowBox[{"2", " ", 
         SuperscriptBox["x", "2"]}], "]"}], 
       RowBox[{"x", "+", "1"}]], ",", 
      RowBox[{"{", 
       RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]}]}], "//", 
  "InputForm"}]], "Input",
 CellLabel->"In[36]:=",
 CellID->514910776],

Cell["0.39934219109623426 + 0.2791805912092563*I", "Output",
 ImageSize->{294, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[36]//InputForm=",
 CellID->29064095]
}, Open  ]],

Cell[TextData[{
 "The value ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " for the option ",
 Cell[BoxData["\"\<FourierFiniteRangeMethod\>\""], "InlineFormula"],
 " means that if the integration strategy specified with the option ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " is one of ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " or ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " then that strategy will be used for the finite range Fourier integration, \
otherwise ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " will be used."
}], "Text",
 CellID->60372860],

Cell[TextData[{
 "Here is a piecewise function integration that uses ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " strategy for the non-oscillatory integral and",
 " ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 " for the finite range oscillatory integral."
}], "MathCaption",
 CellID->617724211],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"\[Piecewise]", GridBox[{
      {
       FractionBox[
        RowBox[{"Cos", "[", 
         RowBox[{"200", " ", "x"}], "]"}], 
        SuperscriptBox["x", "6"]], 
       RowBox[{"0", "<", "x", "<", "20"}]},
      {
       FractionBox["1", 
        SqrtBox[
         RowBox[{"x", "-", "20"}]]], 
       TagBox["True",
        "PiecewiseDefault",
        AutoDelete->False,
        DeletionWarning->True]}
     },
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxItemSize->{
      "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, "Rows" -> {{1.}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.28], {
          Offset[0.84]}, 
         Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "1", ",", "40"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<SymbolicPiecewiseSubdivision\>\"", ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<OscillatorySelection\>\"", ",", 
         RowBox[{"Method", "\[Rule]", "\"\<DoubleExponential\>\""}], ",", 
         RowBox[{"\"\<FourierFiniteRangeMethod\>\"", "\[Rule]", 
          RowBox[{"{", 
           RowBox[{"\"\<LocalAdaptive\>\"", ",", 
            RowBox[{"\"\<Partitioning\>\"", "\[Rule]", "3"}]}], "}"}]}]}], 
        "}"}]}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[37]:=",
 CellID->557648949],

Cell[BoxData["8.948706795921884`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[37]=",
 CellID->382678842]
}, Open  ]],

Cell[TextData[{
 "These are the sampling points of the preceding integration and integral but \
with default option settings. The pattern between ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"[", 
     RowBox[{"0", ",", "20"}], "]"}], " "}], TraditionalForm]], 
  "InlineMath"],
 "on the left picture is typical for the local adaptive \
quadrature\[LongDash]the recursive partitioning into three parts can be seen \
(because of the option ",
 Cell[BoxData[
  RowBox[{"\"\<Partitioning\>\"", "->", "3"}]], "InlineFormula"],
 " given to ",
 Cell[BoxData["\"\<LocalAdaptive\>\""], "InlineFormula"],
 "). The pattern over ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"[", 
     RowBox[{"0", ",", "20"}], "]"}], " "}], TraditionalForm]], 
  "InlineMath"],
 "on the right picture comes from ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 ". The pattern between ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"[", 
     RowBox[{"20", ",", "40"}], "]"}], " "}], TraditionalForm]], 
  "InlineMath"],
 "on the first picture is typical for the double-exponential quadrature. The \
same pattern can be seen on the second picture between ",
 Cell[BoxData[
  FormBox[
   RowBox[{"[", 
    RowBox[{"20", ",", 
     RowBox[{"21", "+", 
      RowBox[{"1", "/", "4"}]}]}], "]"}], TraditionalForm]], "InlineMath"],
 " since ",
 Cell[BoxData["\"\<GlobalAdaptive\>\""], "InlineFormula"],
 " uses by default the ",
 Cell[BoxData["\"\<DoubleExponential\>\""], "InlineFormula"],
 " singularity handler."
}], "MathCaption",
 CellID->437627378],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"pointsDELA", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"\[Piecewise]", GridBox[{
          {
           FractionBox[
            RowBox[{"Cos", "[", 
             RowBox[{"200", " ", "x"}], "]"}], 
            SuperscriptBox["x", "6"]], 
           RowBox[{"0", "<", "x", "<", "20"}]},
          {
           FractionBox["1", 
            SqrtBox[
             RowBox[{"x", "-", "20", " "}]]], 
           TagBox["True",
            "PiecewiseDefault",
            AutoDelete->False,
            DeletionWarning->True]}
         },
         GridBoxAlignment->{
          "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{Baseline}}, "RowsIndexed" -> {}},
         GridBoxItemSize->{
          "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{1.}}, "RowsIndexed" -> {}},
         GridBoxSpacings->{"Columns" -> {
             Offset[0.28], {
              Offset[0.84]}, 
             Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
             Offset[0.2], {
              Offset[0.4]}, 
             Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "1", ",", "40"}], "}"}], ",", 
       RowBox[{"Method", "\[Rule]", 
        RowBox[{"{", 
         RowBox[{"\"\<SymbolicPiecewiseSubdivision\>\"", ",", 
          RowBox[{"Method", "\[Rule]", 
           RowBox[{"{", 
            RowBox[{"\"\<OscillatorySelection\>\"", ",", 
             RowBox[{"Method", "\[Rule]", "\"\<DoubleExponential\>\""}], ",", 
             
             RowBox[{"\"\<FourierFiniteRangeMethod\>\"", "\[Rule]", 
              RowBox[{"{", 
               RowBox[{"\"\<LocalAdaptive\>\"", ",", 
                RowBox[{"\"\<Partitioning\>\"", "\[Rule]", "3"}]}], "}"}]}], 
             ",", 
             RowBox[{
             "\"\<FourierInfiniteRangeMethod\>\"", "\[Rule]", 
              "\"\<ExtrapolatingOscillatory\>\""}]}], "}"}]}]}], "}"}]}], ",",
        
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", 
           RowBox[{"k", "++"}]}], "}"}], "]"}]}]}], "]"}], "]"}], 
    "\[LeftDoubleBracket]", 
    RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{"k", "=", "0"}], ";", 
  RowBox[{"points", "=", 
   RowBox[{
    RowBox[{"Reap", "[", 
     RowBox[{"NIntegrate", "[", 
      RowBox[{
       RowBox[{"\[Piecewise]", GridBox[{
          {
           FractionBox[
            RowBox[{"Cos", "[", 
             RowBox[{"200", " ", "x"}], "]"}], 
            SuperscriptBox["x", "6"]], 
           RowBox[{"0", "<", "x", "<", "20"}]},
          {
           FractionBox["1", 
            RowBox[{
             SqrtBox[
              RowBox[{"x", "-", "20"}]], " "}]], 
           TagBox["True",
            "PiecewiseDefault",
            AutoDelete->False,
            DeletionWarning->True]}
         },
         GridBoxAlignment->{
          "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{Baseline}}, "RowsIndexed" -> {}},
         GridBoxItemSize->{
          "Columns" -> {{Automatic}}, "ColumnsIndexed" -> {}, 
           "Rows" -> {{1.}}, "RowsIndexed" -> {}},
         GridBoxSpacings->{"Columns" -> {
             Offset[0.28], {
              Offset[0.84]}, 
             Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
             Offset[0.2], {
              Offset[0.4]}, 
             Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
       RowBox[{"{", 
        RowBox[{"x", ",", "1", ",", "40"}], "}"}], ",", 
       RowBox[{"EvaluationMonitor", "\[RuleDelayed]", 
        RowBox[{"Sow", "[", 
         RowBox[{"{", 
          RowBox[{"x", ",", 
           RowBox[{"k", "++"}]}], "}"}], "]"}]}]}], "]"}], "]"}], 
    "\[LeftDoubleBracket]", 
    RowBox[{"2", ",", "1"}], "\[RightDoubleBracket]"}]}], ";"}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"grDELA", " ", "=", " ", 
    RowBox[{"Graphics", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"PointSize", "[", "0.01", "]"}], ",", " ", 
        RowBox[{"Point", " ", "/@", " ", "pointsDELA"}]}], "}"}], ",", " ", 
      RowBox[{"AspectRatio", " ", "->", " ", "1"}], ",", " ", 
      RowBox[{"Axes", " ", "->", " ", "True"}], ",", " ", "\n", "    ", 
      RowBox[{"PlotRange", " ", "->", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", " ", "40"}], "}"}], ",", " ", "All"}], "}"}]}]}], 
     "]"}]}], ";"}], " "}], "\n", 
 RowBox[{
  RowBox[{
   RowBox[{"gr", " ", "=", " ", 
    RowBox[{"Graphics", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"PointSize", "[", "0.01", "]"}], ",", " ", 
        RowBox[{"Point", " ", "/@", " ", "points"}]}], "}"}], ",", " ", 
      RowBox[{"AspectRatio", " ", "->", " ", "1"}], ",", " ", 
      RowBox[{"Axes", " ", "->", " ", "True"}], ",", " ", 
      RowBox[{"PlotRange", " ", "->", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"0", ",", " ", "40"}], "}"}], ",", " ", "All"}], "}"}]}]}], 
     "]"}]}], ";"}], " "}], "\n", 
 RowBox[{"GraphicsGrid", "[", 
  RowBox[{"{", 
   RowBox[{"{", 
    RowBox[{"grDELA", ",", "gr"}], "}"}], "}"}], "]"}]}], "Input",
 CellLabel->"In[38]:=",
 CellID->62891410],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNWdtOFEEQHXZ2l5sgEEEI97ug8ZaggqiACCoBNQYefJIQIzwYDfLit/gb
/F873VM123O65rIzu4mbMD3VXVV9qrqqpmY4PL06//bj9Ori7HRm//L01/nF
2e+ZvZ+XwZTf4XnedfD3ZdQL7lVwZ/6ulQrGYtc/wdUPlFSCsRKMncFYBVqv
+0R3aynNoPjHXJ430xJdS/FZkvf8xkzNXEI6ujpcnRYtrdcz96ma6dn/Bs98
Jke2jiwUVTqz0ChLUag7hzfaiWGxtG6tssLzMU11SdLlyOGBdmIo74Hs88nA
1wIPlMGQ7QGTt3Wboy3nlMMP7UeyQCtcU7uAbrZ+l9f11egKny7fae7AUDeJ
OnaseG+Gfme+Cn44dDhqwLGTU9M40A9EzZtsZYJGvX5LxuTGRatmahhP2qfm
UCw6mTNBK0Zjkg9bZEUNMPcBhl6gh2QvWxw3gJ5TsdyLtooiN8FCvT5grTSk
1mms0zhO44ToP452xNllWGzdjbzfB97bKnwSLpv1EVhltLYfD2lOR6VmvRdJ
Gs4qbWR+nwUP2lzh9aGzt0e1wZY9ALpmeT/cLbx5KlrSSTszeu4Ybcs2Y5aw
xFH8pILfdqKvNPnBOjM9mqeJiQe7snG99Wn+NaxzBEwrJXR+ZjYWj56QZQ4d
3L7LkPGBXohbT6hGm95frx80ufcK0D0g350bhS/M7+VAo219LJ6A2dUgkise
Pg1Q+26iJGbzqpJyZija36rBaefgcO3nkmUvcDRjFR02cr0qnlUc3Zxd+vcW
eOqgaYLOybyDhNkV88sx8E+rsHJNR/x68YS0zwM3nsdIYxcrBk6Aa4m0jonc
eFJHQM8RwolIWrL7SMk1HhFPOhiYS0t/hHl+cizG9mbuTzRO2d4y9thv13at
SsvTlwlyaKnezY248PzS+/Ws6iLRW01rRLu4W50pgVCqPMWQ+XReepxtk8/Q
AxulfTjuaJhvEXbsD1qDtZEzxRDabzWbhRBpyUmYb+SnnV9J+dmpXM9IclIF
ttE1KnH0BodvSlCfKjEp6f2Su6MpWJ9LlEqrsFgz44jRvm6gd1K9jVgGiHsX
5t1Ka69yj/5GkHL6rKgCx7to7Bn7aVxL5eYOMt6Hok/sHMiTdfykf9akHsw1
+eSKoNNkN6HaKIlKqAAlfMV14HkBVJqUYwTqksrbgWK9HAB6OadezMkUlM6z
3hdR2m+6KznlOBf6rFHOJ7vvsftTm3uU5lcT+DESOAvWlJTz3Iu+EE5AG/Ek
ktI6hmncsvbSXOui7iHSvQ3zFXvPSPcQja+BW9++EvUPCrY4zxbnfCZp9x1R
J9aKpK8a4Znxt3mpC5WkMHuGhd1seoz0LEcRkv6flTyZjnssJtjANXmU/HVH
CRndgv3xFLEW9BayGqslfwG6a/jTvjAn2aBJjI6epr3B9XKQxvuF8TjRmOLT
3ib1o/f6Ce0jxVGf9R0/CTVmhceRVQId1lPzFSR3ByrlQxHrWH+/iAbrDH9T
yepz5Sr118w2+99tr+MfRGdIhQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 90},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[42]=",
 CellID->241364587]
}, Open  ]],

Cell[TextData[{
 "If the application of a particular oscillatory method is desired for a \
particular type of oscillatory integrals, either the corresponding options of \
",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " should be changed, or ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 "'s ",
 Cell[BoxData[
  ButtonBox["Method",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Method"]], "InlineFormula"],
 " option should be used without the preprocessor ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 "."
}], "Text",
 CellID->95600784],

Cell[TextData[{
 "Here is a piecewise function integration that uses ",
 Cell[BoxData["\"\<ExtrapolatingOscillatory\>\""], "InlineFormula"],
 " for any of the infinite range oscillatory integrals."
}], "MathCaption",
 CellID->352928061],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"\[Piecewise]", GridBox[{
      {
       FractionBox[
        RowBox[{"BesselJ", "[", 
         RowBox[{"3", ",", 
          RowBox[{"-", "x"}]}], "]"}], 
        SqrtBox[
         RowBox[{"-", "x"}]]], 
       RowBox[{"x", "<", "0"}]},
      {
       FractionBox[
        RowBox[{"Cos", "[", 
         RowBox[{"200", " ", "x"}], "]"}], 
        SqrtBox["x"]], 
       RowBox[{"0", "<", "x", "<", "20"}]},
      {
       FractionBox[
        RowBox[{"Sin", "[", 
         RowBox[{
          RowBox[{"2", " ", "x"}], "+", "3"}], "]"}], 
        SuperscriptBox["x", "2"]], 
       RowBox[{"x", ">", "30"}]},
      {
       FractionBox["1", 
        RowBox[{"Log", "[", "x", "]"}]], 
       TagBox["True",
        "PiecewiseDefault",
        AutoDelete->False,
        DeletionWarning->True]}
     },
     GridBoxAlignment->{
      "Columns" -> {{Left}}, "ColumnsIndexed" -> {}, "Rows" -> {{Baseline}}, 
       "RowsIndexed" -> {}},
     GridBoxSpacings->{"Columns" -> {
         Offset[0.28], {
          Offset[0.84]}, 
         Offset[0.28]}, "ColumnsIndexed" -> {}, "Rows" -> {
         Offset[0.2], {
          Offset[0.4]}, 
         Offset[0.2]}, "RowsIndexed" -> {}}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "\[Infinity]"}], ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<SymbolicPiecewiseSubdivision\>\"", ",", 
      RowBox[{"Method", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"\"\<OscillatorySelection\>\"", ",", 
         RowBox[{
         "\"\<FourierInfiniteRangeMethod\>\"", "\[Rule]", 
          "\"\<ExtrapolatingOscillatory\>\""}]}], "}"}]}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->224555532],

Cell[BoxData["3.7793273487232746`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->327845685]
}, Open  ]],

Cell[TextData[{
 "If ",
 Cell[BoxData["\"\<ExtrapolatingOscillatory\>\""], "InlineFormula"],
 " is given as method, ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 " uses it for infinite range oscillatory integration."
}], "MathCaption",
 CellID->178586245],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{
       RowBox[{"2", " ", 
        SuperscriptBox["x", "3"]}], "+", "3"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", "\"\<ExtrapolatingOscillatory\>\""}]}], 
   "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->267033197],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.137979`", ",", 
   RowBox[{"-", "0.020648947228054694`"}]}], "}"}]], "Output",
 ImageSize->{156, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->346237826]
}, Open  ]],

Cell[TextData[{
 "The integration above is faster with the default options of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ". For this integral ",
 Cell[BoxData["\"\<OscillatorySelection\>\""], "InlineFormula"],
 ", which is applied by default, uses ",
 Cell[BoxData["\"\<DoubleExponentialOscillatory\>\""], "InlineFormula"],
 "."
}], "MathCaption",
 CellID->19295388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{"Sin", "[", 
      RowBox[{
       RowBox[{"2", " ", 
        SuperscriptBox["x", "3"]}], "+", "3"}], "]"}], 
     SuperscriptBox["x", "2"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}]}], "]"}], "//", 
  "Timing"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->67214616],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.010997999999999945`", ",", 
   RowBox[{"-", "0.02064894052576892`"}]}], "}"}]], "Output",
 ImageSize->{156, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->176977913]
}, Open  ]],

Cell[CellGroupData[{

Cell["Working with sums of oscillating terms", "Subsubsection",
 CellID->136089857],

Cell[TextData[{
 "In many cases it is useful to apply the oscillatory algorithms to \
integrands that are sums of oscillating functions. That is, each term of such \
integrands is a product of an oscillating function and a less oscillating \
one. (See ",
 ButtonBox["Oscillatory strategies",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/NIntegrateIntegrationStrategies#775797877"],
 " for the forms of recognized as oscillatory by ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 ".)"
}], "Text",
 CellID->442356795],

Cell[TextData[{
 "Here is an example of integration that applies the specialized oscillatory \
",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " algorithms to each term of the integrand."
}], "MathCaption",
 CellID->476726631],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"Cos", "[", 
       RowBox[{"100", " ", "x"}], "]"}], "+", 
      RowBox[{"Sin", "[", 
       RowBox[{"18", " ", "x"}], "]"}], "+", 
      RowBox[{"Cos", "[", 
       RowBox[{"12", " ", "x"}], "]"}]}], 
     SqrtBox["x"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<OscillatorySelection\>\"", ",", 
       RowBox[{"\"\<TermwiseOscillatory\>\"", "\[Rule]", "True"}]}], 
      "}"}]}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->290928299],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.06798900000000006`", ",", "0.08791613468791103`"}], 
  "}"}]], "Output",
 ImageSize->{149, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->238373535]
}, Open  ]],

Cell[TextData[{
 "By default this option is set to ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 StyleBox[", and the integral cannot be computed.", "Text"]
}], "MathCaption",
 CellID->4707458],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "[", 
   RowBox[{
    FractionBox[
     RowBox[{
      RowBox[{"Cos", "[", 
       RowBox[{"100", " ", "x"}], "]"}], "+", 
      RowBox[{"Sin", "[", 
       RowBox[{"18", " ", "x"}], "]"}], "+", 
      RowBox[{"Cos", "[", 
       RowBox[{"12", " ", "x"}], "]"}]}], 
     SqrtBox["x"]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "1", ",", "\[Infinity]"}], "}"}], ",", 
    RowBox[{"Method", "\[Rule]", 
     RowBox[{"{", 
      RowBox[{"\"\<OscillatorySelection\>\"", ",", 
       RowBox[{"\"\<TermwiseOscillatory\>\"", "\[Rule]", "False"}]}], 
      "}"}]}]}], "]"}], "//", "Timing"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->475668539],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.03999399999999983`", ",", "4.834161115541837`*^123"}], 
  "}"}]], "Output",
 ImageSize->{175, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->143096461]
}, Open  ]],

Cell[TextData[{
 "The option is ",
 Cell[BoxData["\"\<TermwiseOscillatory\>\""], "InlineFormula"],
 " is set by default to ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " since splitting the integrals can lead in some cases to divergent \
results."
}], "Text",
 CellID->9882832],

Cell["\<\
Here is a convergent integral which if it is split into two integrals each \
will be divergent.\
\>", "MathCaption",
 CellID->35154748],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Integrate", "[", 
   RowBox[{
    RowBox[{
     FractionBox[
      RowBox[{"Cos", "[", " ", 
       RowBox[{"x", "^", "2"}], "]"}], 
      SqrtBox[
       RowBox[{"1", "-", "x"}]]], "-", 
     FractionBox[
      RowBox[{"Cos", "[", 
       RowBox[{"2", " ", 
        RowBox[{"x", "^", "2"}]}], "]"}], 
      SqrtBox[
       RowBox[{"1", "-", "x"}]]]}], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}], "//", 
  "N"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->69635529],

Cell[BoxData[
 RowBox[{"0.8854221801798188`", "\[InvisibleSpace]", "-", 
  RowBox[{"0.6698632676141547`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{140, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->203938532]
}, Open  ]],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  RowBox[{"\"\<TermwiseOscillatory\>\"", "->", 
   ButtonBox["True",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/True"]}]], "InlineFormula"],
 " is used the result is some big number (and lots of messages). "
}], "MathCaption",
 CellID->542146191],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox[
     RowBox[{"Cos", "[", 
      RowBox[{"x", "^", "2"}], "]"}], 
     SqrtBox[
      RowBox[{"1", "-", "x"}]]], "-", 
    FractionBox[
     RowBox[{"Cos", "[", 
      RowBox[{"2", " ", 
       RowBox[{"x", "^", "2"}]}], "]"}], 
     SqrtBox[
      RowBox[{"1", "-", "x"}]]]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<OscillatorySelection\>\"", ",", 
      RowBox[{"\"\<TermwiseOscillatory\>\"", "\[Rule]", "True"}]}], "}"}]}]}],
   "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->167931815],

Cell[BoxData[
 RowBox[{"0.8103114209690497`", "\[InvisibleSpace]", "-", 
  RowBox[{"3.728226206700831`*^62", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{169, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->457445346]
}, Open  ]],

Cell[TextData[{
 "If ",
 Cell[BoxData[
  RowBox[{"\"\<TermwiseOscillatory\>\"", "->", 
   ButtonBox["False",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/False"]}]], "InlineFormula"],
 " is used (with increased ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 ") the result is closer to the exact one. "
}], "MathCaption",
 CellID->473433075],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    FractionBox[
     RowBox[{"Cos", "[", 
      RowBox[{"x", "^", "2"}], "]"}], 
     SqrtBox[
    