(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     39138,       1317]
NotebookOptionsPosition[     31871,       1082]
NotebookOutlinePosition[     34790,       1156]
CellTagsIndexPosition[     34697,       1150]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Numerical Operations on Functions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/NumericalOperationsOnFunctionsOverview"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Numerical Operations on Functions\"\>"}, "\<\"related \
tutorials\"\>", 
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Integrate" :> Documentation`HelpLookup["paclet:ref/Integrate"],
           "N" :> Documentation`HelpLookup["paclet:ref/N"], "NIntegrate" :> 
          Documentation`HelpLookup["paclet:ref/NIntegrate"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Integrate\"\>", 2->"\<\"N\"\>", 
       3->"\<\"NIntegrate\"\>"}, "\<\"functions\"\>", 
       StyleBox["\<\"functions\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Numerical Integration", "Title",
 CellTags->{"S3.9.3", "3.1"},
 CellID->12104],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["N",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/N"], "[", 
     ButtonBox["Integrate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Integrate"], "[", 
     StyleBox["expr", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["min", "TI"]], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["max", "TI"]], "}", "]", "]"}], Cell["\<\
try to perform an integral exactly, then find numerical approximations to the \
parts that remain\
\>", "TableText"]},
   {
    RowBox[{
     ButtonBox["NIntegrate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/NIntegrate"], "[", 
     StyleBox["expr", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["min", "TI"]], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["max", "TI"]], "}", "]"}], Cell[
    "find a numerical approximation to an integral", "TableText"]},
   {
    RowBox[{
     ButtonBox["NIntegrate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/NIntegrate"], "[", 
     StyleBox["expr", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["min", "TI"]], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], 
      StyleBox["max", "TI"]], "}", ",", "{", 
     StyleBox["y", "TI"], ",", 
     SubscriptBox[
      StyleBox["y", "TI"], 
      StyleBox["min", "TI"]], ",", 
     SubscriptBox[
      StyleBox["y", "TI"], 
      StyleBox["max", "TI"]], "}", ",", 
     StyleBox["\[Ellipsis]", "TR"], "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "multidimensional numerical integral ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubsuperscriptBox["\[Integral]", 
         SubscriptBox["x", 
          StyleBox["min",
           FontSlant->"Italic"]], 
         SubscriptBox["x", 
          StyleBox["max",
           FontSlant->"Italic"]]], 
        RowBox[{"d", "x", " ", 
         RowBox[{
          SubsuperscriptBox["\[Integral]", 
           SubscriptBox["y", 
            StyleBox["min",
             FontSlant->"Italic"]], 
           SubscriptBox["y", 
            StyleBox["max",
             FontSlant->"Italic"]]], 
          StyleBox[
           RowBox[{"dy\[Ellipsis]", 
            StyleBox["expr",
             FontSlant->"Italic"]}]]}]}]}], TraditionalForm]], "InlineMath"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["NIntegrate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/NIntegrate"], "[", 
     RowBox[{
      StyleBox["expr", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["x", "TI"], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["min", "TI"]], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["max", "TI"]]}], "}"}]}], "]"}], "\[SpanFromLeft]"},
   {"", Cell[TextData[{
     "do a numerical integral along a line, starting at ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["x", 
        StyleBox["min", "TI"]], TraditionalForm]], "InlineMath"],
     ", going through the points ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
     ", and ending at ",
     Cell[BoxData[
      FormBox[
       SubscriptBox["x", 
        StyleBox["max", "TI"]], TraditionalForm]], "InlineMath"]
    }], "TableText"]}
  }]], "DefinitionBox",
 CellID->1716],

Cell["Numerical integration functions. ", "Caption",
 CellID->14597],

Cell[TextData[{
 "This finds a numerical approximation to the integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", "0", "\[Infinity]"], 
    SuperscriptBox["e", 
     RowBox[{"-", 
      SuperscriptBox["x", "3"]}]], "\[DifferentialD]", "x"}], 
   TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->13188],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{"x", "^", "3"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7716],

Cell[BoxData["0.8929795115694823`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->797063543]
}, Open  ]],

Cell[TextData[{
 "Here is the numerical value of the double integral ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], "d", "x", " ", 
    SubsuperscriptBox["\[Integral]", 
     RowBox[{"-", "1"}], "1"], "d", "y", " ", 
    RowBox[{"(", 
     RowBox[{
      SuperscriptBox["x", "2"], "+", 
      SuperscriptBox["y", "2"]}], ")"}]}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->28],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], "+", 
    RowBox[{"y", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->6929],

Cell[BoxData["2.6666666666666665`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->334260860]
}, Open  ]],

Cell[TextData[{
 "An important feature of ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " is its ability to deal with functions that \[OpenCurlyDoubleQuote]blow up\
\[CloseCurlyDoubleQuote] at known points. ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " automatically checks for such problems at the end points of the \
integration region. "
}], "Text",
 CellID->23346],

Cell[TextData[{
 "The function ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "/", 
    SqrtBox["x"]}], TraditionalForm]], "InlineMath"],
 " blows up at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0"}], TraditionalForm]], "InlineMath"],
 ", but ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " still succeeds in getting the correct value for the integral. "
}], "MathCaption",
 CellID->3787],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->15094],

Cell[BoxData["2.000000000000003`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->107379572]
}, Open  ]],

Cell[TextData[{
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " can find the integral of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "/", 
    SqrtBox["x"]}], TraditionalForm]], "InlineMath"],
 " exactly. "
}], "MathCaption",
 CellID->2278],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Integrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", "x", "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->23568],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->17365135]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " detects that the singularity in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "/", "x"}], TraditionalForm]], "InlineMath"],
 " at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0"}], TraditionalForm]], "InlineMath"],
 " is not integrable. "
}], "MathCaption",
 CellID->20220],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", "x"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->21891],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[5]:=",
 CellID->328181559],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({1.2241295843469805`*^-225}\\). NIntegrate obtained \
\\!\\(191612.2902185145`\\) and \\!\\(160378.51781028978`\\) for the integral \
and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[5]:=",
 CellID->70531774],

Cell[BoxData["191612.2902185145`"], "Output",
 ImageSize->{56, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->246864406]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " does not automatically look for singularities except at the end points of \
your integration region. When other singularities are present, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " may not give you the right answer for the integral. Nevertheless, in \
following its adaptive procedure, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " will often detect the presence of potentially singular behavior, and will \
warn you about it. "
}], "Text",
 CellID->21089],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " warns you of a possible problem due to the singularity in the middle of \
the integration region. The final result is numerically quite close to the \
correct answer. "
}], "MathCaption",
 CellID->3698],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], " ", 
    RowBox[{"Sin", "[", 
     RowBox[{"1", "/", "x"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->3616],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[6]:=",
 CellID->101105388],

Cell[BoxData["1.3875543584144328`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->3368779]
}, Open  ]],

Cell[TextData[{
 "If you know that your integrand has singularities at particular points, you \
can explicitly tell ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " to deal with them. ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["NIntegrate",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NIntegrate"], "[", 
   RowBox[{
    StyleBox["expr", "TI"], ",", 
    RowBox[{"{", 
     RowBox[{
      StyleBox["x", "TI"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["min", "TI"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["max", "TI"]]}], "}"}]}], "]"}]], "InlineFormula"],
 " integrates ",
 Cell[BoxData[
  StyleBox["expr", "TI"]], "InlineFormula"],
 " from ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    StyleBox["min", "TI"]], TraditionalForm]], "InlineMath"],
 " to ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    StyleBox["max", "TI"]], TraditionalForm]], "InlineMath"],
 ", looking for possible singularities at each of the intermediate points ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 "."
}], "Text",
 CellID->32532],

Cell[TextData[{
 "This gives the same integral, but now explicitly deals with the singularity \
at ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "0"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->16703],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], " ", 
    RowBox[{"Sin", "[", 
     RowBox[{"1", "/", "x"}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->11962],

Cell[BoxData["1.3875544426506425`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->331103196]
}, Open  ]],

Cell[TextData[{
 "You can also use the list of intermediate points ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 " in ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " to specify an integration contour to follow in the complex plane. The \
contour is taken to consist of a sequence of line segments, starting at ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    StyleBox["min", "TI"]], TraditionalForm]], "InlineMath"],
 ", going through each of the ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "i"], TraditionalForm]], "InlineMath"],
 ", and ending at ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", 
    StyleBox["max", "TI"]], TraditionalForm]], "InlineMath"],
 ". "
}], "Text",
 CellID->5596],

Cell[TextData[{
 "This integrates ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "/", "x"}], TraditionalForm]], "InlineMath"],
 " around a closed contour in the complex plane, going from ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "1"}], TraditionalForm]], "InlineMath"],
 ", through the points ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "i"}], TraditionalForm]], "InlineMath"],
 ", 1 and ",
 Cell[BoxData[
  FormBox["i", TraditionalForm]], "InlineMath"],
 ", then back to ",
 Cell[BoxData[
  FormBox[
   RowBox[{"-", "1"}], TraditionalForm]], "InlineMath"],
 ". "
}], "MathCaption",
 CellID->28759],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", "x"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", 
     RowBox[{"-", "I"}], ",", "1", ",", "I", ",", 
     RowBox[{"-", "1"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->8541],

Cell[BoxData[
 RowBox[{"0.`", "\[InvisibleSpace]", "+", 
  RowBox[{"6.283185307179575`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->136820832]
}, Open  ]],

Cell[TextData[{
 "The integral gives ",
 Cell[BoxData[
  FormBox[
   RowBox[{"2", "\[Pi]", "i"}], TraditionalForm]], "InlineMath"],
 ", as expected from Cauchy\[CloseCurlyQuote]s Theorem."
}], "MathCaption",
 CellID->8425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"2", " ", "Pi", " ", "I"}], "]"}]], "Input",
 CellLabel->"In[9]:=",
 CellID->17184],

Cell[BoxData[
 RowBox[{"0.`", "\[InvisibleSpace]", "+", 
  RowBox[{"6.283185307179586`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{91, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->55629499]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    ""},
   {"MinRecursion", "0", Cell[
    "minimum number of recursions for the integration method", "TableText"]},
   {
    ButtonBox["MaxRecursion",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MaxRecursion"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum number of recursions for the integration method", "TableText"]},
   {"MaxPoints", 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "maximum total number of times to sample the integrand", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, False, False, {False}, False}, 
   "RowsIndexed" -> {}},
 GridBoxSpacings->{
  "Columns" -> {0.2800000000000001, {0.56}, 0.2800000000000001}, 
   "ColumnsIndexed" -> {}, "Rows" -> {0.2, {0.16}, 0.2}, 
   "RowsIndexed" -> {}}},
 CellID->8752],

Cell[TextData[{
 "Special options for ",
 StyleBox["NIntegrate", "MR"],
 ". "
}], "Caption",
 CellID->14524],

Cell[TextData[{
 "When ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " tries to evaluate a numerical integral, it samples the integrand at a \
sequence of points. If it finds that the integrand changes rapidly in a \
particular region, then it recursively takes more sample points in that \
region. The parameters ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " specify the minimum and maximum number of recursions to use. Increasing \
the value of ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " guarantees that ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " will use a larger number of sample points. ",
 Cell[BoxData["MaxPoints"], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " limit the number of sample points which ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " will ever try to use. Increasing ",
 Cell[BoxData["MinRecursion"], "InlineFormula"],
 " or ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 " will make ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " work more slowly. "
}], "Text",
 CellID->10018],

Cell[TextData[{
 "With the default settings for all options, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " misses the peak in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"exp", "(", 
    RowBox[{"-", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{"x", "-", "1"}], ")"}], "2"]}], ")"}], TraditionalForm]], 
  "InlineMath"],
 " near ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "1"}], TraditionalForm]], "InlineMath"],
 ", and gives the wrong answer for the integral. "
}], "MathCaption",
 CellID->12093],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", "1"}], ")"}], "^", "2"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1000"}], ",", "1000"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[10]:=",
 CellID->8374],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[10]:=",
 CellID->1103413],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({3.8751667189352395`}\\). NIntegrate obtained \\!\\(1.6330510571683285`\
\\) and \\!\\(0.004736564243403896`\\) for the integral and error estimates. \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", \
ButtonNote -> \\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[10]:=",
 CellID->114104621],

Cell[BoxData["1.6330510571683285`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->777870170]
}, Open  ]],

Cell[TextData[{
 "With the option ",
 Cell[BoxData[
  RowBox[{"MinRecursion", "->", "3"}]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " samples enough points that it notices the peak around ",
 Cell[BoxData[
  FormBox[
   RowBox[{"x", "=", "1"}], TraditionalForm]], "InlineMath"],
 ". With the default setting of ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 ", however, ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " cannot use enough sample points to be able to expect an accurate answer. "
}], "MathCaption",
 CellID->3897],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", "1"}], ")"}], "^", "2"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "50000"}], ",", "1000"}], "}"}], ",", 
   RowBox[{"MinRecursion", "\[Rule]", "3"}]}], "]"}]], "Input",
 CellLabel->"In[41]:=",
 CellID->798],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[41]:=",
 CellID->1286900385],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"ncvb\"\>"}], ":", 
  " ", "\<\"NIntegrate failed to converge to prescribed accuracy after \
\\!\\(9\\) recursive bisections in \\!\\(x\\) near \\!\\({x}\\) = \
\\!\\({\\(-8.445843916606075`\\)}\\). NIntegrate obtained \
\\!\\(1.8181913371063452`\\) and \\!\\(1.165089629798181`\\) for the integral \
and error estimates. \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/ncvb\\\", ButtonNote -> \
\\\"NIntegrate::ncvb\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[41]:=",
 CellID->1626961758],

Cell[BoxData["1.8181913371063452`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[41]=",
 CellID->436989455]
}, Open  ]],

Cell[TextData[{
 "With this setting of ",
 Cell[BoxData[
  ButtonBox["MaxRecursion",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxRecursion"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " can get an accurate answer for the integral. "
}], "MathCaption",
 CellID->27725],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", "1"}], ")"}], "^", "2"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "50000"}], ",", "1000"}], "}"}], ",", 
   RowBox[{"MinRecursion", "\[Rule]", "3"}], ",", 
   RowBox[{"MaxRecursion", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellLabel->"In[42]:=",
 CellID->12372],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"slwcon\"\>"}], ":", 
  " ", "\<\"Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration is 0, highly oscillatory \
integrand, or WorkingPrecision too small. \\!\\(\\*ButtonBox[\\\"\
\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:ref/message/NIntegrate/slwcon\\\", ButtonNote -> \
\\\"NIntegrate::slwcon\\\"]\\)\"\>"}]], "Message", "MSG",
 CellLabel->"During evaluation of In[42]:=",
 CellID->87697445],

Cell[BoxData["1.772418802909317`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[42]=",
 CellID->757176273]
}, Open  ]],

Cell[TextData[{
 "Another way to solve the problem is to make ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " break the integration region into several pieces, with a small piece that \
explicitly covers the neighborhood of the peak. "
}], "MathCaption",
 CellID->29551],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"Exp", "[", 
    RowBox[{"-", 
     RowBox[{
      RowBox[{"(", 
       RowBox[{"x", "-", "1"}], ")"}], "^", "2"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1000"}], ",", 
     RowBox[{"-", "10"}], ",", "10", ",", "1000"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[13]:=",
 CellID->13557],

Cell[BoxData["1.7724538509093477`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[13]=",
 CellID->154017907]
}, Open  ]],

Cell[TextData[{
 "For integrals in many dimensions, it can take a long time for ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " to get a precise answer. However, by setting the option ",
 Cell[BoxData["MaxPoints"], "InlineFormula"],
 ", you can tell ",
 Cell[BoxData[
  ButtonBox["NIntegrate",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NIntegrate"]], "InlineFormula"],
 " to give you just a rough estimate, sampling the integrand only a limited \
number of times. "
}], "Text",
 CellID->13491],

Cell["\<\
Here is a way to get a rough estimate for an integral that takes a long time \
to compute. \
\>", "MathCaption",
 CellID->26861],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NIntegrate", "[", 
  RowBox[{
   RowBox[{"1", "/", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"x", "+", 
      RowBox[{
       RowBox[{"Log", "[", 
        RowBox[{"y", "+", "z"}], "]"}], "^", "2"}]}], "]"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"y", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"MaxPoints", "->", "1000"}]}], "]"}]], "Input",
 CellLabel->"In[14]:=",
 CellID->22988],

Cell[BoxData[
 RowBox[{
  RowBox[{"NIntegrate", "::", "\<\"maxp\"\>"}], ":", 
  " ", "\<\"The integral failed to converge after \\!\\(1023\\) integrand \
evaluations. NIntegrate obtained \\!\\(1.4548878649546768`\\) and \
\\!\\(0.03247010762528413`\\) for the integral and error estimates.\"\>"}]], \
"Message", "MSG",
 CellLabel->"During evaluation of In[14]:=",
 CellID->796721450],

Cell[BoxData["1.4548878649546768`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[14]=",
 CellID->190141403]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->303562147],

Cell[TextData[ButtonBox["Numerical Operations on Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/NumericalOperationsOnFunctionsOverview"]], \
"RelatedTutorials",
 CellID->163046772]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Numerical Integration - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, "Overview" -> {"toc" -> Cell[
      BoxData[
       ButtonBox[
        StyleBox["Numerical Operations on Functions", "OverviewNavText"], 
        BaseStyle -> "Link", ButtonData -> 
        "paclet:tutorial/NumericalOperationsOnFunctionsOverview"]], "Text", 
      FontFamily -> "Verdana"], "prevnext" -> Cell[
      TextData[{
        ButtonBox[
         StyleBox["\[FilledLeftTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/TheUncertaintiesOfNumericalMathematics"], 
        "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]", 
        ButtonBox[
         StyleBox["\[FilledRightTriangle]", "OverviewNavText"], BaseStyle -> 
         "Link", ButtonData -> 
         "paclet:tutorial/NumericalEvaluationOfSumsAndProducts"]}], "Text", 
      FontFamily -> "Verdana"]}, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 41, 15.0110382}", "context" -> "", 
    "keywords" -> {
     "Cauchy's theorem", "Cauchy theorem", "integration", 
      "Las Vegas integration", "MaxPoints", "MaxRecursion", "MinRecursion", 
      "Monte Carlo integration", "NIntegrate", "numerical integration", 
      "quadrature", "singularities"}, "index" -> True, "label" -> 
    "Mathematica Tutorial", "language" -> "en", "paclet" -> "Mathematica", 
    "status" -> "None", "summary" -> 
    "Numerical integration functions. This finds a numerical approximation to \
the integral SubsuperscriptBox[\\[Integral], 0, Infinity]e^-x^3dx. Here is \
the numerical value of the double integral SubsuperscriptBox[\\[Integral], \
RowBox[{-, 1}], 1]dx SubsuperscriptBox[\\[Integral], RowBox[{-, 1}], 1]dy \
(x^2+y^2). ", "synonyms" -> {"3.9.3"}, "title" -> "Numerical Integration", 
    "type" -> "Tutorial", "uri" -> "tutorial/NumericalIntegration"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "S3.9.3"->{
  Cell[2027, 61, 84, 2, 70, "Title",
   CellTags->{"S3.9.3", "3.1"},
   CellID->12104]},
 "3.1"->{
  Cell[2027, 61, 84, 2, 70, "Title",
   CellTags->{"S3.9.3", "3.1"},
   CellID->12104]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"S3.9.3", 34486, 1139},
 {"3.1", 34585, 1143}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1402, 34, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2027, 61, 84, 2, 70, "Title",
 CellTags->{"S3.9.3", "3.1"},
 CellID->12104],
Cell[2114, 65, 3732, 121, 70, "DefinitionBox",
 CellID->1716],
Cell[5849, 188, 68, 1, 70, "Caption",
 CellID->14597],
Cell[5920, 191, 355, 12, 70, "MathCaption",
 CellID->13188],
Cell[CellGroupData[{
Cell[6300, 207, 267, 9, 28, "Input",
 CellID->7716],
Cell[6570, 218, 176, 5, 36, "Output",
 CellID->797063543]
}, Open  ]],
Cell[6761, 226, 464, 15, 70, "MathCaption",
 CellID->28],
Cell[CellGroupData[{
Cell[7250, 245, 368, 13, 28, "Input",
 CellID->6929],
Cell[7621, 260, 176, 5, 36, "Output",
 CellID->334260860]
}, Open  ]],
Cell[7812, 268, 532, 15, 70, "Text",
 CellID->23346],
Cell[8347, 285, 476, 17, 70, "MathCaption",
 CellID->3787],
Cell[CellGroupData[{
Cell[8848, 306, 241, 8, 70, "Input",
 CellID->15094],
Cell[9092, 316, 175, 5, 36, "Output",
 CellID->107379572]
}, Open  ]],
Cell[9282, 324, 245, 10, 70, "MathCaption",
 CellID->2278],
Cell[CellGroupData[{
Cell[9552, 338, 240, 8, 70, "Input",
 CellID->23568],
Cell[9795, 348, 157, 5, 36, "Output",
 CellID->17365135]
}, Open  ]],
Cell[9967, 356, 420, 15, 70, "MathCaption",
 CellID->20220],
Cell[CellGroupData[{
Cell[10412, 375, 208, 7, 70, "Input",
 CellID->21891],
Cell[10623, 384, 566, 10, 70, "Message",
 CellID->328181559],
Cell[11192, 396, 665, 12, 70, "Message",
 CellID->70531774],
Cell[11860, 410, 175, 5, 36, "Output",
 CellID->246864406]
}, Open  ]],
Cell[12050, 418, 768, 20, 70, "Text",
 CellID->21089],
Cell[12821, 440, 344, 9, 70, "MathCaption",
 CellID->3698],
Cell[CellGroupData[{
Cell[13190, 453, 311, 11, 70, "Input",
 CellID->3616],
Cell[13504, 466, 566, 10, 70, "Message",
 CellID->101105388],
Cell[14073, 478, 174, 5, 36, "Output",
 CellID->3368779]
}, Open  ]],
Cell[14262, 486, 1429, 50, 70, "Text",
 CellID->32532],
Cell[15694, 538, 230, 8, 70, "MathCaption",
 CellID->16703],
Cell[CellGroupData[{
Cell[15949, 550, 322, 11, 70, "Input",
 CellID->11962],
Cell[16274, 563, 176, 5, 36, "Output",
 CellID->331103196]
}, Open  ]],
Cell[16465, 571, 832, 27, 70, "Text",
 CellID->5596],
Cell[17300, 600, 600, 22, 70, "MathCaption",
 CellID->28759],
Cell[CellGroupData[{
Cell[17925, 626, 300, 10, 70, "Input",
 CellID->8541],
Cell[18228, 638, 255, 7, 36, "Output",
 CellID->136820832]
}, Open  ]],
Cell[18498, 648, 222, 7, 70, "MathCaption",
 CellID->8425],
Cell[CellGroupData[{
Cell[18745, 659, 127, 4, 70, "Input",
 CellID->17184],
Cell[18875, 665, 254, 7, 36, "Output",
 CellID->55629499]
}, Open  ]],
Cell[19144, 675, 1080, 28, 70, "DefinitionBox3Col",
 CellID->8752],
Cell[20227, 705, 108, 5, 70, "Caption",
 CellID->14524],
Cell[20338, 712, 1685, 50, 70, "Text",
 CellID->10018],
Cell[22026, 764, 596, 21, 70, "MathCaption",
 CellID->12093],
Cell[CellGroupData[{
Cell[22647, 789, 343, 12, 70, "Input",
 CellID->8374],
Cell[22993, 803, 565, 10, 70, "Message",
 CellID->1103413],
Cell[23561, 815, 662, 11, 70, "Message",
 CellID->114104621],
Cell[24226, 828, 177, 5, 36, "Output",
 CellID->777870170]
}, Open  ]],
Cell[24418, 836, 800, 25, 70, "MathCaption",
 CellID->3897],
Cell[CellGroupData[{
Cell[25243, 865, 394, 13, 70, "Input",
 CellID->798],
Cell[25640, 880, 568, 10, 70, "Message",
 CellID->1286900385],
Cell[26211, 892, 668, 12, 70, "Message",
 CellID->1626961758],
Cell[26882, 906, 177, 5, 36, "Output",
 CellID->436989455]
}, Open  ]],
Cell[27074, 914, 383, 13, 70, "MathCaption",
 CellID->27725],
Cell[CellGroupData[{
Cell[27482, 931, 448, 14, 70, "Input",
 CellID->12372],
Cell[27933, 947, 566, 10, 70, "Message",
 CellID->87697445],
Cell[28502, 959, 176, 5, 36, "Output",
 CellID->757176273]
}, Open  ]],
Cell[28693, 967, 351, 9, 70, "MathCaption",
 CellID->29551],
Cell[CellGroupData[{
Cell[29069, 980, 387, 13, 70, "Input",
 CellID->13557],
Cell[29459, 995, 177, 5, 36, "Output",
 CellID->154017907]
}, Open  ]],
Cell[29651, 1003, 581, 16, 70, "Text",
 CellID->13491],
Cell[30235, 1021, 138, 4, 70, "MathCaption",
 CellID->26861],
Cell[CellGroupData[{
Cell[30398, 1029, 543, 17, 70, "Input",
 CellID->22988],
Cell[30944, 1048, 383, 8, 70, "Message",
 CellID->796721450],
Cell[31330, 1058, 177, 5, 36, "Output",
 CellID->190141403]
}, Open  ]],
Cell[CellGroupData[{
Cell[31544, 1068, 72, 1, 70, "RelatedTutorialsSection",
 CellID->303562147],
Cell[31619, 1071, 198, 5, 70, "RelatedTutorials",
 CellID->163046772]
}, Open  ]]
}, Open  ]],
Cell[31844, 1080, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

