(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     71317,       2054]
NotebookOptionsPosition[     56565,       1578]
NotebookOutlinePosition[     65264,       1830]
CellTagsIndexPosition[     65118,       1822]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["MATHEMATICA TUTORIAL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Unconstrained Optimization" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/UnconstrainedOptimizationOverview"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Unconstrained Optimization\"\>"}, "\<\"related tutorials\"\>", 
       
       StyleBox["\<\"related tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Optimization" :> 
          Documentation`HelpLookup["paclet:guide/Optimization"], 
          "Matrix-Based Minimization" :> 
          Documentation`HelpLookup["paclet:guide/MatrixBasedMinimization"], 
          "Curve Fitting & Approximate Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/CurveFittingAndApproximateFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Optimization\"\>", 
       2->"\<\"Matrix-Based Minimization\"\>", 
       3->"\<\"Curve Fitting & Approximate Functions\"\>"}, "\<\"more about\"\
\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Termination Conditions", "Title",
 CellTags->{"t:24", "c:24", "b:0.5.3"},
 CellID->1953968666],

Cell[TextData[{
 "Mathematically, sufficient conditions for a local minimum of a smooth \
function are quite straightforward: ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["x", "*"], TraditionalForm]]],
 "is a local minimum if ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[Del]", 
     RowBox[{"f", "(", 
      SuperscriptBox["x", "*"], ")"}]}], " ", "=", " ", "0"}], 
   TraditionalForm]]],
 " and the Hessian ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["\[Del]", "2"], 
    RowBox[{"f", "(", 
     SuperscriptBox["x", "*"], ")"}]}], TraditionalForm]]],
 " is positive definite. (It is a necessary condition that the Hessian be \
positive semidefinite.) The conditions for a root are even simpler.",
 " ",
 "However, when the function f is being evaluated on a computer where its \
value is only known, at best, to a certain precision, and practically only a \
limited number of function evaluations are possible, it is necessary to use \
error estimates to decide when a search has become close enough to a minimum \
or a root, and to compute the solution only to a finite tolerance. For the \
most part, these estimates suffice quite well, but in some cases, they can be \
in error, usually due to unresolved fine scale behavior of the function."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->1031439538],

Cell["\<\
Tolerances affect how close a search will try to get to a root or local \
minimum before terminating the search. Assuming that the function itself has \
some error (as is typical when it is computed with numerical values), it is \
not typically possible to locate the position of a minimum much better than \
to half of the precision of the numbers being worked with. This is because of \
the quadratic nature of local minima. Near the bottom of a parabola, the \
height varies quite slowly as you move across from the minimum. Thus, if \
there is any error noise in the function, it will typically mask the actual \
rise of the parabola over a width roughly equal to the square root of the \
noise. This is best seen with an example,\
\>", "Text",
 CellTags->"b:0.5.3",
 CellID->916316582],

Cell[TextData[StyleBox["This loads a package that contains some utility \
functions:",
 FontSize->11,
 FontWeight->"Plain"]], "MathCaption",
 CellID->18098],

Cell[BoxData[
 RowBox[{"<<", "Optimization`UnconstrainedProblems`"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23546],

Cell[TextData[{
 "The following command displays a sequence of plots showing the minimum of \
the function ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"sin", "(", "x", ")"}], "-", 
    RowBox[{"cos", 
     RowBox[{"(", "x", ")"}]}], "+", 
    SqrtBox["2"]}], TraditionalForm]], "InlineMath"],
 " over successively smaller ranges.",
 " ",
 "The curve computed with machine numbers is shown in black; the actual curve \
(computed with 100 digits of precision) is shown in blue."
}], "MathCaption",
 CellTags->"b:0.5.3",
 CellID->927561037],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Block", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"\[Epsilon]", "=", 
       RowBox[{"10.", "^", 
        RowBox[{"-", "k"}]}]}], "}"}], ",", 
     RowBox[{"Show", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Plot", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"Sin", "[", "x", "]"}], "-", 
            RowBox[{"Cos", "[", "x", "]"}], "+", 
            RowBox[{"Sqrt", "[", "2", "]"}]}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", 
             RowBox[{
              RowBox[{
               RowBox[{"-", "\[Pi]"}], "/", "4"}], "-", "\[Epsilon]"}], ",", 
             RowBox[{
              RowBox[{
               RowBox[{"-", "\[Pi]"}], "/", "4"}], "+", "\[Epsilon]"}]}], 
            "}"}], ",", 
           RowBox[{"PlotStyle", "\[Rule]", "Black"}]}], "]"}], ",", 
         RowBox[{"Plot", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"Sin", "[", "x", "]"}], "-", 
            RowBox[{"Cos", "[", "x", "]"}], "+", 
            RowBox[{"Sqrt", "[", "2", "]"}]}], ",", 
           RowBox[{"{", 
            RowBox[{"x", ",", 
             RowBox[{
              RowBox[{
               RowBox[{"-", "\[Pi]"}], "/", "4"}], "-", "\[Epsilon]"}], ",", 
             RowBox[{
              RowBox[{
               RowBox[{"-", "\[Pi]"}], "/", "4"}], "+", "\[Epsilon]"}]}], 
            "}"}], ",", 
           RowBox[{"PlotStyle", "\[Rule]", "Blue"}], ",", 
           RowBox[{"WorkingPrecision", "\[Rule]", "100"}]}], "]"}]}], "}"}], 
       ",", 
       RowBox[{"PlotLabel", "\[Rule]", 
        RowBox[{"Row", "[", 
         RowBox[{"{", 
          RowBox[{"\"\<Width \>\"", ",", "\[Epsilon]"}], "}"}], "]"}]}]}], 
      "]"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "5", ",", "9"}], "}"}]}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[11]:=",
 CellID->934555286],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{577, 265},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[11]=",
 CellID->460277261]
}, Open  ]],

Cell[TextData[{
 "From the sequence of plots, it is clear that for changes of order ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SuperscriptBox["10", 
     RowBox[{"-", "8"}]], ","}], TraditionalForm]]],
 "which is about half of machine precision and smaller, errors in the \
function are masking the actual shape of the curve near the minimum. With \
just sampling of the function at that precision, there is no way to be sure \
if a given point gives the smallest local value of the function or not to any \
closer tolerance."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->2524937],

Cell["\<\
The value of the derivative, if it is computed symbolically, is much more \
reliable, but for the general case, it is not sufficient to rely only on the \
value of the derivative: the search needs to find a local minimal value of \
the function where the derivative is small to satisfy the tolerances in \
general. Note also that if symbolic derivatives of your function cannot be \
computed and finite differences or a derivative free method is used, the \
accuracy of the solution may degrade further.\
\>", "Text",
 CellTags->"b:0.5.3",
 CellID->1634426866],

Cell["\<\
Root finding can suffer from the same inaccuracies in the function. While it \
is typically not as severe, some of the error estimates are based on a merit \
function, which does have a quadratic shape.\
\>", "Text",
 CellTags->"b:0.5.3",
 CellID->1349008085],

Cell[TextData[{
 "For the reason of this limitation, the default tolerances for the ",
 StyleBox["Find", "MR"],
 " functions are all set to be half of the final working precision. Depending \
on how much error the function has, this may or may not be achievable, but in \
most cases it is a reasonable goal. You can adjust the tolerances using the \
",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " options. When ",
 StyleBox["AccuracyGoal", "MR"],
 "->",
 StyleBox["ag", "TI"],
 " and ",
 StyleBox["PrecisionGoal", "MR"],
 "->",
 StyleBox["pg", "TI"],
 ", this defines tolerances ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["tol", "a"], " ", "=", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "ag"}]]}], TraditionalForm]]],
 "and ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["tol", "r"], " ", "=", " ", 
    SuperscriptBox["10", 
     RowBox[{"-", "pg"}]]}], TraditionalForm]]],
 "."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->1794023216],

Cell[TextData[{
 "Given ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["tol", "a"], TraditionalForm]]],
 " and ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["tol", "r"], TraditionalForm]]],
 " ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " tries to find a value ",
 Cell[BoxData[
  FormBox[
   SubscriptBox["x", "k"], TraditionalForm]]],
 "such that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[LeftDoubleBracketingBar]", 
     RowBox[{
      SubscriptBox["x", "k"], " ", "-", " ", 
      SuperscriptBox["x", "*"]}], "\[RightDoubleBracketingBar]"}], " ", 
    "\[LessEqual]", " ", 
    RowBox[{"max", "(", 
     RowBox[{
      SubscriptBox["tol", "a"], ",", " ", 
      RowBox[{
       RowBox[{"\[LeftDoubleBracketingBar]", 
        SubscriptBox["x", "k"], "\[RightDoubleBracketingBar]"}], " ", 
       SubscriptBox["tol", "r"]}]}], ")"}]}], TraditionalForm]]],
 ". Of course, since the exact position of the minimum, ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["x", "*"], TraditionalForm]]],
 "is not known, the quantity ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[LeftDoubleBracketingBar]", 
    RowBox[{
     SubscriptBox["x", "k"], " ", "-", " ", 
     SuperscriptBox["x", "*"]}], "\[RightDoubleBracketingBar]"}], 
   TraditionalForm]]],
 " is estimated. This is usually done based on past steps and derivative \
values. To match the derivative condition at a minimum, the additional \
requirement ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"\[LeftDoubleBracketingBar]", 
     RowBox[{"\[Del]", 
      RowBox[{"f", "(", 
       SubscriptBox["x", "k"], ")"}]}], "\[RightDoubleBracketingBar]"}], 
    "\[LessEqual]", " ", 
    SubscriptBox["tol", "a"]}], TraditionalForm]]],
 " is imposed. For ",
 Cell[BoxData[
  ButtonBox["FindRoot",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindRoot"]], "InlineFormula"],
 ", the corresponding condition is that just the residual be small at the \
root: ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
    "\[LeftDoubleBracketingBar]", "f", "\[RightDoubleBracketingBar]"}], 
    "\[LessEqual]", " ", 
    SubscriptBox["tol", "a"]}], TraditionalForm]]],
 "."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->1477787432],

Cell[TextData[{
 "This finds the ",
 Cell[BoxData[
  FormBox[
   SqrtBox["2"], TraditionalForm]]],
 "to at least 12 digits of accuracy, or within a tolerance of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", 
    RowBox[{"-", "12"}]], TraditionalForm]]],
 ". The precision goal of \[Infinity] means that ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["tol", "r"], " ", "=", " ", "0"}], TraditionalForm]]],
 ", so it does not have any effect in the formula.",
 " ",
 "(Note, you cannot similarly set the accuracy goal to \[Infinity] since that \
is always used for the size of the residual.)"
}], "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1524992069],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindRoot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "2"}], " ", "-", " ", "2"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "1"}], "}"}], ",", " ", 
   RowBox[{"AccuracyGoal", "\[Rule]", "12"}], ",", " ", 
   RowBox[{"PrecisionGoal", "\[Rule]", "\[Infinity]"}]}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[33]:=",
 CellID->324376738],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"x", "\[Rule]", "1.4142135623730951`"}], "}"}]], "Output",
 ImageSize->{90, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[33]=",
 CellID->1142510972]
}, Open  ]],

Cell["\<\
This shows that the result satisfied the requested error tolerances.\
\>", "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1377858456],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"{", 
   RowBox[{
    RowBox[{"x", " ", "-", " ", 
     RowBox[{"Sqrt", "[", "2", "]"}]}], ",", " ", 
    RowBox[{
     RowBox[{"x", "^", "2"}], " ", "-", " ", "2"}]}], "}"}], " ", "/.", " ", 
  "%"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[34]:=",
 CellID->1350904391],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.`", ",", "4.440892098500626`*^-16"}], "}"}]], "Output",
 ImageSize->{133, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[34]=",
 CellID->1530618577]
}, Open  ]],

Cell[TextData[{
 "This tries to find the minimum of the function ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"sin", "(", "x", ")"}], "-", 
    RowBox[{"cos", "(", "x", ")"}]}], TraditionalForm]], "InlineMath"],
 " to 8 digits of accuracy.",
 " ",
 Cell[BoxData[
  ButtonBox["FindMinimum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindMinimum"]], "InlineFormula"],
 " gives a warning message because of the error in the function as seen in \
the plots."
}], "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1256344534],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], " ", "-", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0"}], "}"}], ",", " ", 
   RowBox[{"Method", "->", "\"\<Newton\>\""}], ",", " ", 
   RowBox[{"AccuracyGoal", "\[Rule]", "8"}], ",", " ", 
   RowBox[{"PrecisionGoal", "\[Rule]", "\[Infinity]"}]}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[35]:=",
 CellID->1323365634],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"lstol\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"The line search decreased the step \
size to within tolerance specified by AccuracyGoal and PrecisionGoal but was \
unable to find a sufficient decrease in the function.  You may need more than \
\\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(MachinePrecision\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" digits \
of working precision to meet these tolerances.\\\\\\\"\\\", \\\"MSG\\\"]\\) \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, \
ButtonData:>\\\"paclet:Messages/FindMinimum/lstol\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellTags->"b:0.5.3",
 CellLabel->"During evaluation of In[35]:=",
 CellID->399816226],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.414213562373095`"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "0.7853981759996976`"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{187, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[35]=",
 CellID->234978919]
}, Open  ]],

Cell[TextData[{
 "This shows that though the value at the minimum was found to be basically \
machine epsilon, the position was only found to the order of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["10", 
    RowBox[{"-", "8"}]], TraditionalForm]]],
 " or so."
}], "MathCaption",
 CellTags->"b:0.5.3",
 CellID->559214230],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], " ", "+", " ", 
    RowBox[{"%", "[", 
     RowBox[{"[", "1", "]"}], "]"}]}], ",", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"\[Pi]", "/", "4"}], " ", "+", " ", "x"}], " ", "/.", " ", 
    RowBox[{"%", "[", 
     RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[36]:=",
 CellID->1325124338],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2.220446049250313`*^-16", ",", 
   RowBox[{"-", "1.2602249355175843`*^-8"}]}], "}"}]], "Output",
 ImageSize->{212, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[36]=",
 CellID->68264800]
}, Open  ]],

Cell[TextData[{
 "In multiple dimensions, the situation is even more complicated since there \
can be more error in some directions than others, such as when a minimum is \
found along a relatively narrow valley, as in the ",
 ButtonBox["Freudenstein Roth problem",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/UnconstrainedOptimizationTestProblems"],
 ". For searches such as this, often the search parameters are scaled, which \
in turn affects the error estimates.",
 " ",
 "Nonetheless, it is still typical that the quadratic shape of the minimum \
affects the realistically achievable tolerances."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->934579334],

Cell[TextData[{
 "When you need to find a root or minimum beyond the default tolerances, it \
may be necessary to increase the final working precision. You can do this \
with the ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 " option. When you use ",
 StyleBox["WorkingPrecision->", "MR"],
 StyleBox["prec", "TI"],
 ", the search starts at the precision of the starting values and is \
adaptively increased up to ",
 StyleBox["prec", "TI"],
 " as the search converges. By default, ",
 StyleBox["WorkingPrecision->MachinePrecision", "MR"],
 ", so machine numbers are used, which are usually much faster. Going to \
higher precision can take significantly more time, but can get you much more \
accurate results if your function is defined in an appropriate way.",
 " ",
 "For very high precision solutions, ",
 ButtonBox["Newton\[CloseCurlyQuote]s",
  BaseStyle->"Link",
  ButtonData->
   "paclet:tutorial/UnconstrainedOptimizationNewtonsMethodMinimum"],
 " method is recommended because its quadratic convergence rate significantly \
reduces the number of steps ultimately required."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->779472380],

Cell[TextData[{
 "It is important to note that increasing the setting of the ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 " option does no good if the function is defined with lower precision \
numbers. In general, for ",
 StyleBox["WorkingPrecision->", "MR"],
 StyleBox["prec", "TI"],
 " to be effective, the numbers used to define the function should be exact \
or at least of precision ",
 StyleBox["prec", "TI"],
 ". When possible, the precision of numbers in the function is artificially \
raised to ",
 StyleBox["prec", "TI"],
 " using ",
 Cell[BoxData[
  ButtonBox["SetPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SetPrecision"]], "InlineFormula"],
 " so that convergence still works, but this is not always possible. In any \
case, when the functions and derivatives are evaluated numerically, the \
precision of the results is raised to ",
 StyleBox["prec", "TI"],
 " if necessary so that the internal arithmetic can be done with ",
 StyleBox["prec", "TI"],
 " digit precision. Even so, the actual precision or accuracy of the root or \
minimum and its position is limited by the accuracy in the function.",
 " ",
 "This is especially important to keep in mind when using ",
 Cell[BoxData[
  ButtonBox["FindFit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/FindFit"]], "InlineFormula"],
 " where data is usually only known up to a certain precision."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->942043960],

Cell["Here is a function defined using machine numbers.", "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1683390036],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"f", "[", 
    RowBox[{"x_", "?", "NumberQ"}], "]"}], " ", ":=", " ", 
   RowBox[{
    RowBox[{"Sin", "[", 
     RowBox[{"1.", " ", "x"}], "]"}], " ", "-", " ", 
    RowBox[{"Cos", "[", 
     RowBox[{"1.", " ", "x"}], "]"}]}]}], ";"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[37]:=",
 CellID->1786769474],

Cell["\<\
Even with higher working precision, the minimum cannot be resolved better \
because the actual function still has the same errors as shown in the plots. \
The derivatives were specified to keep other things consistent with the \
computation at machine precision shown previously.\
\>", "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1970713517],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{"f", "[", "x", "]"}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0"}], "}"}], ",", 
   RowBox[{"Gradient", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Cos", "[", 
       RowBox[{"1.", " ", "x"}], "]"}], " ", "+", " ", 
      RowBox[{"Sin", "[", 
       RowBox[{"1.", " ", "x"}], "]"}]}], "}"}]}], ",", " ", 
   RowBox[{"Method", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{"\"\<Newton\>\"", ",", " ", 
      RowBox[{"Hessian", "\[Rule]", 
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Cos", "[", 
           RowBox[{"1.", " ", "x"}], "]"}], " ", "-", " ", 
          RowBox[{"Sin", "[", 
           RowBox[{"1.", " ", "x"}], "]"}]}], "}"}], "}"}]}]}], "}"}]}], ",", 
   " ", 
   RowBox[{"AccuracyGoal", "\[Rule]", "8"}], ",", " ", 
   RowBox[{"PrecisionGoal", "\[Rule]", "\[Infinity]"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[38]:=",
 CellID->1141792538],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"lstol\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"The line search decreased the step \
size to within tolerance specified by AccuracyGoal and PrecisionGoal but was \
unable to find a sufficient decrease in the function.  You may need more than \
\\\\\\\"\\\", \\\"MSG\\\"]\\)\\!\\(20.`\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" \
digits of working precision to meet these tolerances.\\\\\\\"\\\", \
\\\"MSG\\\"]\\) \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:Messages/FindMinimum/lstol\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellTags->"b:0.5.3",
 CellLabel->"During evaluation of In[38]:=",
 CellID->570226102],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.41421356237309492343001693370752036572`20."}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
     "-", "0.78539817599970194669315678201701162664`20.000000000000007"}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{383, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[38]=",
 CellID->1718063400]
}, Open  ]],

Cell["\<\
Here is the computation done with 20 digit precision when the function does \
not have machine numbers.\
\>", "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1192658128],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], " ", "-", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0"}], "}"}], ",", " ", 
   RowBox[{"Method", "->", "\"\<Newton\>\""}], ",", " ", 
   RowBox[{"AccuracyGoal", "\[Rule]", "8"}], ",", " ", 
   RowBox[{"PrecisionGoal", "\[Rule]", "\[Infinity]"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "20"}]}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[39]:=",
 CellID->392796198],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "1.41421356237309504880168872420969807857`20."}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{"-", "0.78539816339744830961566017867003129032`20."}]}], "}"}]}],
   "}"}]], "Output",
 ImageSize->{383, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[39]=",
 CellID->426446238]
}, Open  ]],

Cell[TextData[{
 "If you specify ",
 StyleBox["WorkingPrecision->", "MR"],
 StyleBox["prec,", "TI"],
 " but do not explicitly specify the ",
 Cell[BoxData[
  ButtonBox["AccuracyGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AccuracyGoal"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["PrecisionGoal",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/PrecisionGoal"]], "InlineFormula"],
 " options, then their default settings of ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " will be taken to be ",
 StyleBox["AccuracyGoal->", "MR"],
 StyleBox["prec", "TI"],
 StyleBox["/2", "MR"],
 " and ",
 StyleBox["PrecisionGoal->", "MR"],
 StyleBox["prec", "TI"],
 StyleBox["/2", "MR"],
 ".",
 " ",
 "This leads to the smallest tolerances which can realistically be expected \
in general, as discussed earlier."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->163133981],

Cell[TextData[{
 "Here is the computation done with 50 digit precision without an explicitly \
specified setting for the ",
 StyleBox["AccuracyGoal", "MR"],
 " or ",
 StyleBox["PrecisionGoal", "MR"],
 " options."
}], "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1603685095],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"FindMinimum", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sin", "[", "x", "]"}], " ", "-", " ", 
    RowBox[{"Cos", "[", "x", "]"}]}], ",", " ", 
   RowBox[{"{", 
    RowBox[{"x", ",", " ", "0"}], "}"}], ",", " ", 
   RowBox[{"Method", "->", "\"\<Newton\>\""}], ",", " ", 
   RowBox[{"WorkingPrecision", "\[Rule]", "50"}]}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[40]:=",
 CellID->68138168],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
   "-", "1.4142135623730950488016887242096980785696718753769480731766797379907\
`50."}], ",", 
   RowBox[{"{", 
    RowBox[{"x", "\[Rule]", 
     RowBox[{
     "-", "0.78539816339744830961566084581987572104929234984377645524373615577\
97`50."}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{427, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[40]=",
 CellID->1128992027]
}, Open  ]],

Cell["\<\
This shows that though the value at the minimum was actually found to be even \
better than the default 25 digit tolerances.\
\>", "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1947440111],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], " ", "+", " ", 
    RowBox[{"%", "[", 
     RowBox[{"[", "1", "]"}], "]"}]}], ",", " ", 
   RowBox[{
    RowBox[{
     RowBox[{"\[Pi]", "/", "4"}], " ", "+", " ", "x"}], " ", "/.", " ", 
    RowBox[{"%", "[", 
     RowBox[{"[", "2", "]"}], "]"}]}]}], "}"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[41]:=",
 CellID->827681650],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0``49.84948500216801", ",", "0``50.104910118633825"}], 
  "}"}]], "Output",
 ImageSize->{139, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[41]=",
 CellID->1950976544]
}, Open  ]],

Cell["\<\
The following table below shows a summary of the options affecting precision \
and tolerance.\
\>", "Text",
 CellTags->"b:0.5.3",
 CellID->1133197639],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    ""},
   {
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[TextData[{
     "The final working precision, ",
     Cell[BoxData[
      StyleBox["prec", "TI"]], "InlineFormula"],
     ", to use. Precision is adaptively increased from the smaller of ",
     Cell[BoxData[
      StyleBox["prec", "TI"]], "InlineFormula"],
     " and the precision of the starting conditions to",
     " ",
     Cell[BoxData[
      StyleBox["prec", "TI"]], "InlineFormula"],
     ". "
    }], "TableText"]},
   {
    ButtonBox["AccuracyGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/AccuracyGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "Setting ",
     Cell[BoxData[
      StyleBox["ag", "TI"]], "InlineFormula"],
     " determines an absolute tolerance by ",
     Cell[BoxData[
      RowBox[{
       SubscriptBox[
        StyleBox["tol", "TI"], 
        StyleBox["a", "TI"]], "=", 
       SuperscriptBox["10", 
        RowBox[{"-", 
         StyleBox["ag", "TI"]}]]}]], "InlineFormula"],
     ".",
     " ",
     "When ",
     Cell[BoxData[
      ButtonBox["Automatic",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      RowBox[{
       StyleBox["ag", "TI"], "=", 
       RowBox[{
        StyleBox["prec", "TI"], "/", "2"}]}]], "InlineFormula"],
     "."
    }], "TableText"]},
   {
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[TextData[{
     "Setting ",
     Cell[BoxData[
      StyleBox["pg", "TI"]], "InlineFormula"],
     " determines an absolute tolerance by ",
     Cell[BoxData[
      RowBox[{
       SubscriptBox[
        StyleBox["tol", "TI"], 
        StyleBox["r", "TI"]], "=", 
       SuperscriptBox["10", 
        RowBox[{"-", 
         StyleBox["pg", "TI"]}]]}]], "InlineFormula"],
     ". When ",
     Cell[BoxData[
      ButtonBox["Automatic",
       BaseStyle->"Link",
       ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      RowBox[{
       StyleBox["pg", "TI"], "=", 
       RowBox[{
        StyleBox["prec", "TI"], "/", "2"}]}]], "InlineFormula"],
     "."
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {False, 
     AbsoluteThickness[0.5], {False}, False}, "RowsIndexed" -> {}}},
 CellTags->"b:0.2.3",
 CellID->204030633],

Cell[TextData[{
 "Precision and tolerance options in the ",
 Cell[BoxData["\"\<Find\>\""], "InlineFormula"],
 " functions. "
}], "Caption",
 CellTags->"b:0.5.3",
 CellID->2068280541],

Cell[TextData[{
 "It is sometimes the case that a search will converge slowly. To prevent \
slow searches from going on indefinitely, the ",
 StyleBox["Find", "MR"],
 " commands all have a maximum number of iterations (steps) that will be \
allowed before terminating. This can be controlled with the option ",
 Cell[BoxData[
  ButtonBox["MaxIterations",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/MaxIterations"]], "InlineFormula"],
 ", that has the default value ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["MaxIterations",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MaxIterations"], "->", "100"}]], 
  "InlineFormula"],
 ".",
 " ",
 "When a search terminates with this condition, the command will issue the ",
 StyleBox["cvmit", "MR"],
 " message."
}], "Text",
 CellTags->"b:0.5.3",
 CellID->273258598],

Cell[TextData[{
 "This gets the Brown Dennis problem from the ",
 StyleBox[ButtonBox["Optimization`UnconstrainedProblems`",
  BaseStyle->"Link",
  ButtonData->"paclet:tutorial/UnconstrainedOptimizationTestProblems"], 
  "MR"],
 " package."
}], "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1337219300],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Short", "[", 
  RowBox[{
   RowBox[{"bd", " ", "=", " ", 
    RowBox[{"GetFindMinimumProblem", "[", "BrownDennis", "]"}]}], ",", " ", 
   "5"}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[42]:=",
 CellID->1228409245],

Cell[BoxData[
 TagBox[
  RowBox[{"FindMinimumProblem", "[", 
   RowBox[{
    RowBox[{
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            SuperscriptBox["\[ExponentialE]", 
             RowBox[{"1", "/", "5"}]]}], "+", 
           SubscriptBox["X", "1"], "+", 
           FractionBox[
            SubscriptBox["X", "2"], "5"]}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            RowBox[{"Cos", "[", 
             FractionBox["1", "5"], "]"}]}], "+", 
           SubscriptBox["X", "3"], "+", 
           RowBox[{
            RowBox[{"Sin", "[", 
             FractionBox["1", "5"], "]"}], " ", 
            SubscriptBox["X", "4"]}]}], ")"}], "2"]}], ")"}], "2"], "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            SuperscriptBox["\[ExponentialE]", 
             RowBox[{"2", "/", "5"}]]}], "+", 
           SubscriptBox["X", "1"], "+", 
           FractionBox[
            RowBox[{"2", " ", 
             SubscriptBox["X", "2"]}], "5"]}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            RowBox[{"Cos", "[", 
             FractionBox["2", "5"], "]"}]}], "+", 
           SubscriptBox["X", "3"], "+", 
           RowBox[{
            RowBox[{"Sin", "[", 
             FractionBox["2", "5"], "]"}], " ", 
            SubscriptBox["X", "4"]}]}], ")"}], "2"]}], ")"}], "2"], "+", 
     RowBox[{"\[LeftSkeleton]", "17", "\[RightSkeleton]"}], "+", 
     SuperscriptBox[
      RowBox[{"(", 
       RowBox[{
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            SuperscriptBox["\[ExponentialE]", "4"]}], "+", 
           SubscriptBox["X", "1"], "+", 
           RowBox[{"4", " ", 
            SubscriptBox["X", "2"]}]}], ")"}], "2"], "+", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{
           RowBox[{"-", 
            RowBox[{"Cos", "[", "4", "]"}]}], "+", 
           SubscriptBox["X", "3"], "+", 
           RowBox[{
            RowBox[{"Sin", "[", "4", "]"}], " ", 
            SubscriptBox["X", "4"]}]}], ")"}], "2"]}], ")"}], "2"]}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["X", "1"], ",", "25.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["X", "2"], ",", "5.`"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["X", "3"], ",", 
        RowBox[{"-", "5.`"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox["X", "4"], ",", 
        RowBox[{"-", "1.`"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", "}"}], ",", "BrownDennis", ",", 
    RowBox[{"{", 
     RowBox[{"4", ",", "20"}], "}"}]}], "]"}],
  Short[#, 5]& ]], "Output",
 ImageSize->{466, 107},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[42]//Short=",
 CellID->134968965]
}, Open  ]],

Cell["\<\
This attempt solves the problem with the default method, which is the \
Levenberg-Marquardt method, since the function is a sum of squares.\
\>", "MathCaption",
 CellTags->"b:0.5.3",
 CellID->47065960],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ProblemSolve", "[", "bd", "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[43]:=",
 CellID->763607624],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"cvmit\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"Failed to converge to the requested \
accuracy or precision within \\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(100\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" iterations.\\\\\\\"\
\\\", \\\"MSG\\\"]\\) \\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", \
ButtonStyle->\\\"Link\\\", ButtonFrame->None, \
ButtonData:>\\\"paclet:Messages/FindMinimum/cvmit\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellTags->"b:0.5.3",
 CellLabel->"During evaluation of In[43]:=",
 CellID->680663523],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"105443.33642522906`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["X", "1"], "\[Rule]", 
      RowBox[{"-", "7.350709157306787`"}]}], ",", 
     RowBox[{
      SubscriptBox["X", "2"], "\[Rule]", "11.736486214847858`"}], ",", 
     RowBox[{
      SubscriptBox["X", "3"], "\[Rule]", 
      RowBox[{"-", "0.6043601375170637`"}]}], ",", 
     RowBox[{
      SubscriptBox["X", "4"], "\[Rule]", "0.1683957949845426`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{461, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[43]=",
 CellID->266443492]
}, Open  ]],

Cell["\<\
The Levenberg Marquardt method is converging slowly on this problem because \
the residual is nonzero near the minimum and the second-order part of the \
Hessian is needed. While the method eventually does converge in just under \
400 steps, perhaps a better option is to use a method which may converge \
faster.\
\>", "Text",
 CellTags->"b:0.5.3",
 CellID->1638752904],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ProblemSolve", "[", 
  RowBox[{"bd", ",", " ", 
   RowBox[{"Method", "\[Rule]", "QuasiNewton"}]}], "]"}]], "Input",
 CellTags->"b:0.5.3",
 CellLabel->"In[44]:=",
 CellID->1042284753],

Cell[BoxData[
 RowBox[{
  RowBox[{"FindMinimum", "::", "\<\"lstol\"\>"}], ":", 
  " ", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"The line search decreased the step \
size to within tolerance specified by AccuracyGoal and PrecisionGoal but was \
unable to find a sufficient decrease in the function.  You may need more than \
\\\\\\\"\\\", \
\\\"MSG\\\"]\\)\\!\\(MachinePrecision\\)\\!\\(\\*StyleBox[\\\"\\\\\\\" digits \
of working precision to meet these tolerances.\\\\\\\"\\\", \\\"MSG\\\"]\\) \
\\!\\(\\*ButtonBox[\\\"\[RightSkeleton]\\\", ButtonStyle->\\\"Link\\\", \
ButtonFrame->None, \
ButtonData:>\\\"paclet:Messages/FindMinimum/lstol\\\"]\\)\"\>"}]], "Message", \
"MSG",
 CellTags->"b:0.5.3",
 CellLabel->"During evaluation of In[44]:=",
 CellID->612366049],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"85822.20162635628`", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      SubscriptBox["X", "1"], "\[Rule]", 
      RowBox[{"-", "11.594439915244358`"}]}], ",", 
     RowBox[{
      SubscriptBox["X", "2"], "\[Rule]", "13.203630055464703`"}], ",", 
     RowBox[{
      SubscriptBox["X", "3"], "\[Rule]", 
      RowBox[{"-", "0.40343949495004133`"}]}], ",", 
     RowBox[{
      SubscriptBox["X", "4"], "\[Rule]", "0.23677878064059513`"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{468, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellTags->"b:0.5.3",
 CellLabel->"Out[44]=",
 CellID->1294549119]
}, Open  ]],

Cell["\<\
In a larger calculation, one possibility when hitting the iteration limit is \
to use the final search point, which is returned, as a starting condition for \
continuing the search, ideally with another method.\
\>", "Text",
 CellTags->"b:0.5.3",
 CellID->53614840],

Cell[CellGroupData[{

Cell["MORE ABOUT", "TutorialMoreAboutSection",
 CellID->118428827],

Cell[TextData[ButtonBox["Optimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/Optimization"]], "TutorialMoreAbout",
 CellID->48539711],

Cell[TextData[ButtonBox["Matrix-Based Minimization",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/MatrixBasedMinimization"]], "TutorialMoreAbout",
 CellID->21321],

Cell[TextData[ButtonBox["Curve Fitting & Approximate Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/CurveFittingAndApproximateFunctions"]], "TutorialMoreAbout",
 CellID->776025961]
}, Open  ]],

Cell[CellGroupData[{

Cell["RELATED TUTORIALS", "RelatedTutorialsSection",
 CellID->5365388],

Cell[TextData[ButtonBox["Unconstrained Optimization",
 BaseStyle->"Link",
 ButtonData->
  "paclet:tutorial/UnconstrainedOptimizationOverview"]], "RelatedTutorials",
 CellID->20683583]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Termination Conditions - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 20, 45, 24.2517936}", "context" -> 
    "Optimization`UnconstrainedProblems`", "keywords" -> {}, "index" -> True, 
    "label" -> "Mathematica Tutorial", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "Mathematically, sufficient conditions for a local minimum of a smooth \
function are quite straightforward: x^*is a local minimum if \\[Del]f(x^*) = \
0 and the Hessian \\[Del]^2f(x^*) is positive definite. (It is a necessary \
condition that the Hessian be positive semidefinite.) The conditions for a \
root are even simpler. However, when the function f is being evaluated on a \
computer where its value is only known, at best, to a certain precision, and \
practically only a limited number of function evaluations are possible, it is \
necessary to use error estimates to decide when a search has become close \
enough to a minimum or a root, and to compute the solution only to a finite \
tolerance. For the most part, these estimates suffice quite well, but in some \
cases, they can be in error, usually due to unresolved fine scale behavior of \
the function. Tolerances affect how close a search will try to get to a root \
or local minimum before terminating the search. Assuming that the function \
itself has some error (as is typical when it is computed with numerical \
values), it is not typically possible to locate the position of a minimum \
much better than to half of the precision of the numbers being worked with. \
This is because of the quadratic nature of local minima. Near the bottom of a \
parabola, the height varies quite slowly as you move across from the minimum. \
Thus, if there is any error noise in the function, it will typically mask the \
actual rise of the parabola over a width roughly equal to the square root of \
the noise. This is best seen with an example, This loads a package that \
contains some utility functions:", "synonyms" -> {}, "title" -> 
    "Termination Conditions", "type" -> "Tutorial", "uri" -> 
    "tutorial/UnconstrainedOptimizationTerminationConditions"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "t:24"->{
  Cell[2234, 67, 100, 2, 70, "Title",
   CellTags->{"t:24", "c:24", "b:0.5.3"},
   CellID->1953968666]},
 "c:24"->{
  Cell[2234, 67, 100, 2, 70, "Title",
   CellTags->{"t:24", "c:24", "b:0.5.3"},
   CellID->1953968666]},
 "b:0.5.3"->{
  Cell[2234, 67, 100, 2, 70, "Title",
   CellTags->{"t:24", "c:24", "b:0.5.3"},
   CellID->1953968666],
  Cell[2337, 71, 1331, 33, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->1031439538],
  Cell[3671, 106, 800, 13, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->916316582],
  Cell[4754, 132, 546, 16, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->927561037],
  Cell[5325, 152, 1950, 56, 83, "Input",
   CellTags->"b:0.5.3",
   CellID->934555286],
  Cell[7278, 210, 15473, 258, 286, "Output",
   Evaluatable->False,
   CellTags->"b:0.5.3",
   CellID->460277261],
  Cell[22766, 471, 580, 14, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->2524937],
  Cell[23349, 487, 570, 10, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->1634426866],
  Cell[23922, 499, 269, 6, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->1349008085],
  Cell[24194, 507, 1193, 41, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->1794023216],
  Cell[25390, 550, 2268, 75, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->1477787432],
  Cell[27661, 627, 670, 21, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1524992069],
  Cell[28356, 652, 392, 11, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->324376738],
  Cell[28751, 665, 251, 8, 36, "Output",
   CellTags->"b:0.5.3",
   CellID->1142510972],
  Cell[29017, 676, 142, 4, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1377858456],
  Cell[29184, 684, 312, 11, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->1350904391],
  Cell[29499, 697, 252, 8, 39, "Output",
   CellTags->"b:0.5.3",
   CellID->1530618577],
  Cell[29766, 708, 531, 17, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1256344534],
  Cell[30322, 729, 492, 13, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->1323365634],
  Cell[30817, 744, 760, 15, 70, "Message",
   CellTags->"b:0.5.3",
   CellID->399816226],
  Cell[31580, 761, 357, 12, 36, "Output",
   CellTags->"b:0.5.3",
   CellID->234978919],
  Cell[31952, 776, 325, 10, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->559214230],
  Cell[32302, 790, 423, 14, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->1325124338],
  Cell[32728, 806, 289, 9, 39, "Output",
   CellTags->"b:0.5.3",
   CellID->68264800],
  Cell[33032, 818, 663, 14, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->934579334],
  Cell[33698, 834, 1230, 29, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->779472380],
  Cell[34931, 865, 1528, 38, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->942043960],
  Cell[36462, 905, 115, 2, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1683390036],
  Cell[36580, 909, 358, 12, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->1786769474],
  Cell[36941, 923, 353, 7, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1970713517],
  Cell[37319, 934, 1056, 30, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->1141792538],
  Cell[38378, 966, 746, 14, 70, "Message",
   CellTags->"b:0.5.3",
   CellID->570226102],
  Cell[39127, 982, 435, 14, 36, "Output",
   CellTags->"b:0.5.3",
   CellID->1718063400],
  Cell[39577, 999, 177, 5, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1192658128],
  Cell[39779, 1008, 547, 14, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->392796198],
  Cell[40329, 1024, 411, 13, 36, "Output",
   CellTags->"b:0.5.3",
   CellID->426446238],
  Cell[40755, 1040, 955, 33, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->163133981],
  Cell[41713, 1075, 274, 9, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1603685095],
  Cell[42012, 1088, 428, 12, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->68138168],
  Cell[42443, 1102, 481, 16, 54, "Output",
   CellTags->"b:0.5.3",
   CellID->1128992027],
  Cell[42939, 1121, 198, 5, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1947440111],
  Cell[43162, 1130, 422, 14, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->827681650],
  Cell[43587, 1146, 270, 9, 39, "Output",
   CellTags->"b:0.5.3",
   CellID->1950976544],
  Cell[43872, 1158, 160, 5, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->1133197639],
  Cell[46901, 1261, 182, 6, 70, "Caption",
   CellTags->"b:0.5.3",
   CellID->2068280541],
  Cell[47086, 1269, 817, 24, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->273258598],
  Cell[47906, 1295, 301, 9, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->1337219300],
  Cell[48232, 1308, 252, 8, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->1228409245],
  Cell[48487, 1318, 3153, 98, 142, "Output",
   CellTags->"b:0.5.3",
   CellID->134968965],
  Cell[51655, 1419, 211, 5, 70, "MathCaption",
   CellTags->"b:0.5.3",
   CellID->47065960],
  Cell[51891, 1428, 132, 4, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->763607624],
  Cell[52026, 1434, 574, 12, 70, "Message",
   CellTags->"b:0.5.3",
   CellID->680663523],
  Cell[52603, 1448, 664, 21, 36, "Output",
   CellTags->"b:0.5.3",
   CellID->266443492],
  Cell[53282, 1472, 380, 8, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->1638752904],
  Cell[53687, 1484, 206, 6, 70, "Input",
   CellTags->"b:0.5.3",
   CellID->1042284753],
  Cell[53896, 1492, 760, 15, 70, "Message",
   CellTags->"b:0.5.3",
   CellID->612366049],
  Cell[54659, 1509, 667, 21, 36, "Output",
   CellTags->"b:0.5.3",
   CellID->1294549119],
  Cell[55341, 1533, 275, 6, 70, "Text",
   CellTags->"b:0.5.3",
   CellID->53614840]},
 "b:0.2.3"->{
  Cell[44035, 1165, 2863, 94, 70, "DefinitionBox3Col",
   CellTags->"b:0.2.3",
   CellID->204030633]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"t:24", 59485, 1631},
 {"c:24", 59601, 1635},
 {"b:0.5.3", 59720, 1639},
 {"b:0.2.3", 64992, 1815}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1609, 40, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2234, 67, 100, 2, 70, "Title",
 CellTags->{"t:24", "c:24", "b:0.5.3"},
 CellID->1953968666],
Cell[2337, 71, 1331, 33, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->1031439538],
Cell[3671, 106, 800, 13, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->916316582],
Cell[4474, 121, 156, 4, 70, "MathCaption",
 CellID->18098],
Cell[4633, 127, 118, 3, 70, "Input",
 CellID->23546],
Cell[4754, 132, 546, 16, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->927561037],
Cell[CellGroupData[{
Cell[5325, 152, 1950, 56, 83, "Input",
 CellTags->"b:0.5.3",
 CellID->934555286],
Cell[7278, 210, 15473, 258, 286, "Output",
 Evaluatable->False,
 CellTags->"b:0.5.3",
 CellID->460277261]
}, Open  ]],
Cell[22766, 471, 580, 14, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->2524937],
Cell[23349, 487, 570, 10, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->1634426866],
Cell[23922, 499, 269, 6, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->1349008085],
Cell[24194, 507, 1193, 41, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->1794023216],
Cell[25390, 550, 2268, 75, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->1477787432],
Cell[27661, 627, 670, 21, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1524992069],
Cell[CellGroupData[{
Cell[28356, 652, 392, 11, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->324376738],
Cell[28751, 665, 251, 8, 36, "Output",
 CellTags->"b:0.5.3",
 CellID->1142510972]
}, Open  ]],
Cell[29017, 676, 142, 4, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1377858456],
Cell[CellGroupData[{
Cell[29184, 684, 312, 11, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->1350904391],
Cell[29499, 697, 252, 8, 39, "Output",
 CellTags->"b:0.5.3",
 CellID->1530618577]
}, Open  ]],
Cell[29766, 708, 531, 17, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1256344534],
Cell[CellGroupData[{
Cell[30322, 729, 492, 13, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->1323365634],
Cell[30817, 744, 760, 15, 70, "Message",
 CellTags->"b:0.5.3",
 CellID->399816226],
Cell[31580, 761, 357, 12, 36, "Output",
 CellTags->"b:0.5.3",
 CellID->234978919]
}, Open  ]],
Cell[31952, 776, 325, 10, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->559214230],
Cell[CellGroupData[{
Cell[32302, 790, 423, 14, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->1325124338],
Cell[32728, 806, 289, 9, 39, "Output",
 CellTags->"b:0.5.3",
 CellID->68264800]
}, Open  ]],
Cell[33032, 818, 663, 14, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->934579334],
Cell[33698, 834, 1230, 29, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->779472380],
Cell[34931, 865, 1528, 38, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->942043960],
Cell[36462, 905, 115, 2, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1683390036],
Cell[36580, 909, 358, 12, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->1786769474],
Cell[36941, 923, 353, 7, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1970713517],
Cell[CellGroupData[{
Cell[37319, 934, 1056, 30, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->1141792538],
Cell[38378, 966, 746, 14, 70, "Message",
 CellTags->"b:0.5.3",
 CellID->570226102],
Cell[39127, 982, 435, 14, 36, "Output",
 CellTags->"b:0.5.3",
 CellID->1718063400]
}, Open  ]],
Cell[39577, 999, 177, 5, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1192658128],
Cell[CellGroupData[{
Cell[39779, 1008, 547, 14, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->392796198],
Cell[40329, 1024, 411, 13, 36, "Output",
 CellTags->"b:0.5.3",
 CellID->426446238]
}, Open  ]],
Cell[40755, 1040, 955, 33, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->163133981],
Cell[41713, 1075, 274, 9, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1603685095],
Cell[CellGroupData[{
Cell[42012, 1088, 428, 12, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->68138168],
Cell[42443, 1102, 481, 16, 54, "Output",
 CellTags->"b:0.5.3",
 CellID->1128992027]
}, Open  ]],
Cell[42939, 1121, 198, 5, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1947440111],
Cell[CellGroupData[{
Cell[43162, 1130, 422, 14, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->827681650],
Cell[43587, 1146, 270, 9, 39, "Output",
 CellTags->"b:0.5.3",
 CellID->1950976544]
}, Open  ]],
Cell[43872, 1158, 160, 5, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->1133197639],
Cell[44035, 1165, 2863, 94, 70, "DefinitionBox3Col",
 CellTags->"b:0.2.3",
 CellID->204030633],
Cell[46901, 1261, 182, 6, 70, "Caption",
 CellTags->"b:0.5.3",
 CellID->2068280541],
Cell[47086, 1269, 817, 24, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->273258598],
Cell[47906, 1295, 301, 9, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->1337219300],
Cell[CellGroupData[{
Cell[48232, 1308, 252, 8, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->1228409245],
Cell[48487, 1318, 3153, 98, 142, "Output",
 CellTags->"b:0.5.3",
 CellID->134968965]
}, Open  ]],
Cell[51655, 1419, 211, 5, 70, "MathCaption",
 CellTags->"b:0.5.3",
 CellID->47065960],
Cell[CellGroupData[{
Cell[51891, 1428, 132, 4, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->763607624],
Cell[52026, 1434, 574, 12, 70, "Message",
 CellTags->"b:0.5.3",
 CellID->680663523],
Cell[52603, 1448, 664, 21, 36, "Output",
 CellTags->"b:0.5.3",
 CellID->266443492]
}, Open  ]],
Cell[53282, 1472, 380, 8, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->1638752904],
Cell[CellGroupData[{
Cell[53687, 1484, 206, 6, 70, "Input",
 CellTags->"b:0.5.3",
 CellID->1042284753],
Cell[53896, 1492, 760, 15, 70, "Message",
 CellTags->"b:0.5.3",
 CellID->612366049],
Cell[54659, 1509, 667, 21, 36, "Output",
 CellTags->"b:0.5.3",
 CellID->1294549119]
}, Open  ]],
Cell[55341, 1533, 275, 6, 70, "Text",
 CellTags->"b:0.5.3",
 CellID->53614840],
Cell[CellGroupData[{
Cell[55641, 1543, 66, 1, 70, "TutorialMoreAboutSection",
 CellID->118428827],
Cell[55710, 1546, 143, 3, 70, "TutorialMoreAbout",
 CellID->48539711],
Cell[55856, 1551, 164, 3, 70, "TutorialMoreAbout",
 CellID->21321],
Cell[56023, 1556, 195, 4, 70, "TutorialMoreAbout",
 CellID->776025961]
}, Open  ]],
Cell[CellGroupData[{
Cell[56255, 1565, 70, 1, 70, "RelatedTutorialsSection",
 CellID->5365388],
Cell[56328, 1568, 183, 4, 70, "RelatedTutorials",
 CellID->20683583]
}, Open  ]]
}, Open  ]],
Cell[56538, 1576, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

