(* :Context: AuthorTools`Bookmarks` *)

(* :Author: Louis J. D'Andria *)

(* :Summary:
    Managing bookmarks to information in the Help
    Browser and elsewhere.
*)

(* :Copyright: *)

(* :Package Version: $Revision: 1.38 $ $Date: 2004/10/12 19:28:32 $ *)

(* :Mathematica Version: 6.0 *)

(* :History:

*)

(* :Keywords:
     
*)

(* :Discussion:

      
*)

(* :Warning:
    
*)




BeginPackage["AuthorTools`Bookmarks`"]


BookmarkCategory::usage = "BookmarkCategory is a structure that contains bookmark data.";

Bookmark::usage = "Bookmark[name, location, address] represents a stored bookmark.";



AddBookmark::usage = "AddBookmark[b] adds the given bookmark expression to the user's bookmarks.";

BookmarkCurrentPage::usage = "BookmarkCurrentPage[] adds a bookmark to the current Help Browser page to the user's bookmarks.";

BookmarkNotebook::usage = "BookmarkNotebook[lis] returns a Notebook expression containing links to the given bookmarks.";

CategoryList;

GetBookmarkData::usage = "GetBookmarkData[] returns a list of the user's bookmarks.";

HelpBrowserNotebookQ::usage = "HelpBrowserNotebookQ[nbObj] returns True if the given notebook object is a Help Browser notebook, and False otherwise.";

RemoveBookmark::usage = "RemoveBookmark[b] removes the given bookmarm from the user's bookmarks.";

ShowBookmarks::usage = "ShowBookmarks[] shows the user's bookmarks in a palette window.";



WriteBookmarkData::usage = "WriteBookmarkData[file, lis] writes the given data to the file.";

WriteBookmarkNotebook::usage = "WriteBookmarkNotebook[] rewrites the bookmark notebook based on the current bookmark data.";

ImportBookmarks::usage = "ImportBookmarks[file] imports the bookmarks in the given data file into the user's listing.";


$BookmarkDataFile::usage = "$BookmarkDataFile is the location of the stored bookmark data file.";

$BookmarkDirectory::usage = "$BookmarkDirectory is the location of the bookmark data file, categories file, etc.";

$BookmarkNotebookFile::usage = "$BookmarkNotebookFile is the location of the bookmark notebook file used by the Help Browser.";

InitializeBookmarks::usage = "InitializeBookmarks[] puts default bookmark data and browser categories in $BookmarkDirectory.";






Begin["`Private`"]


(* Possible features:


Filter bookmarks by Help Browser categories
 (Item Lookup filtering bookmarks?)

Sort by type of bookmark (Help Browser, file, web)
Sort by last visited date
Sort alphabetically by name
Sort alphabetically by URL


Rename bookmarks
Remove bookmarks

Check validity of bookmarks - whether target exists


Import bookmarks from other Mathematicas
Export bookmarks from other Mathematicas
Import bookmarks from HTML
Export bookmarks to HTML


Copy bookmark 'URL'

Put some bookmarks on the toolbar
Put some bookmarks into the menus

Open Help Browser location in the Help Browser
Open Help Browser location in original notebook (??)
Open Help Browser location in new untitled notebook (NotebookLookup...)
Open Help Browser location in new Help Browser


Convert Help Browser location to web location (documents.wolfram.com/v5/...)


How about subscribing to WRI RSS feeds, or getting notebook content
pushed from WRI???


*)



(*

   BookmarkCategory["WRI",
     {
       Bookmark["Somewhere offline", "Help", {"AddOnsLink", "Interface Introduction"}],
       Bookmark["Somewhere online", "Web", "http://www.wolfram.com"],
       Bookmark["Some file somewhere", "File", "/Users/joe/file.nb"]
       BookmarkCategory["More...",
         {
           Bookmark["more", "Web", "http://www.mathworld.com"]
         }
       ]
     }
   ]
   
   How about:
   "LastVisited" -> Date[] | AbsoluteTime[]
   "Toolbar" -> True | False -- maybe this should just be a "Toolbar" category instead
   "Menu" -> True | False -- again, just have a "Menu" category instead?
   
*)



$BookmarkDirectory = ToFileName[{$UserBaseDirectory, "Autoload", "Bookmarks"}];

$BookmarkDataFile = ToFileName[{$BookmarkDirectory}, "BookmarkData.m"];

$BookmarkNotebookFile = ToFileName[{$BookmarkDirectory}, "Bookmarks.nb"];




stripEmptyCategories[lis_] := lis //. BookmarkCategory[_,{}] :> Sequence[]



GetBookmarkData[] := GetBookmarkData[$BookmarkDataFile]

GetBookmarkData[file_String] := 
Block[{lis},
  If[FileType[file] =!= File, Return @ {}];
  lis = Get[file];
  If[Not @ ListQ @ lis, Return @ {}];
  stripEmptyCategories[lis]
]




AddBookmark[b_Bookmark, {cats___}] := 
Block[{lis = GetBookmarkData[]},
  lis = addBookmark[lis, b, {cats}];
  WriteBookmarkData[$BookmarkDataFile, lis];
  (* WriteBookmarkNotebook[lis]; disable for now *)
]

addBookmark[{args___}, b_Bookmark, {}|{""}] := {args, b};

addBookmark[{x___, BookmarkCategory[c_, lis_], y___}, b_Bookmark, {c_, cats___}] :=
  {x, BookmarkCategory[c, addBookmark[lis, b, {cats}]], y}

addBookmark[{args___}, b_Bookmark, {c_, cats___}] :=
  {args, BookmarkCategory[c, addBookmark[{}, b, {cats}]]}



(*
   FIXME: this must be improved so that when duplicate bookmarks
   are found, it only removes the one the user wanted to remove.
*)

RemoveBookmark::dup = "More than one copy of bookmark found -- removing all. `1`";

RemoveBookmark[b_Bookmark, cats_] := 
Block[{lis = GetBookmarkData[], newlis},
  If[ Count[lis, b, Infinity] > 1, Message[RemoveBookmarks::dup, b]];
  newlis = DeleteCases[lis, b, Infinity];
  
  If[newlis === lis, Return[]];
  newlis = stripEmptyCategories[newlis];
  WriteBookmarkData[$BookmarkDataFile, newlis];
  (* WriteBookmarkNotebook[newlis]; disable for now *)
]








WriteBookmarkNotebook[] := WriteBookmarkNotebook[GetBookmarkData[$BookmarkDataFile]];

WriteBookmarkNotebook[lis_] := WriteBookmarkNotebook[$BookmarkNotebookFile, lis];

WriteBookmarkNotebook[f_, lis_] :=  Export[f, BookmarkNotebook[lis], "Notebook"];






ShowBookmarksInNotebook[nbObj_NotebookObject] :=
Block[{nbExpr},
  nbExpr = BookmarkNotebook[GetBookmarkData[]];
  NotebookPut[nbExpr, nbObj];
  SelectionMove[nbObj, Before, Notebook];
  SetSelectedNotebook[nbobj];
  Export[$BookmarkNotebookFile, nbExpr, "Notebook"];
  nbObj
]





ShowBookmarks[] := 
Block[{nbExpr, nbObj},

  If[(nbObj = ButtonNotebook[]) =!= $Failed, Return @ ShowBookmarksInNotebook @ nbObj];
    
  nbExpr = Join[BookmarkNotebook[GetBookmarkData[]], Notebook[
    WindowFrame -> "Normal", (* for editing *)
    WindowClickSelect -> True, (* for edting *)
    WindowSize -> {275, Automatic},
    WindowElements -> {"VerticalScrollBar"}
  ]];
  
  If[MemberQ[Notebooks[], $BookmarkNotebook],
    NotebookPut[nbExpr, $BookmarkNotebook];
    SelectionMove[$BookmarkNotebook, Before, Notebook]
    ,
    $BookmarkNotebook = NotebookPut[nbExpr]
  ];
  $BookmarkNotebook
]





BookmarkNotebook[lis_] :=
Block[{},
  
  Notebook[Flatten @ {
      
      Cell["Bookmarks", "Section"],
      
      BookmarkCurrentPageCell[],

      horizontalRule,
      
      If[lis === {},
        Cell["None", "Text"],
        recursiveFormatBookmarks[lis, {}]
      ],
      
	  newBookmarkCell[],

      (*
      openCopyCell[],
      *)
      
      horizontalRule
    },
    StyleDefinitions -> "HelpBrowser.nb",
    PageWidth -> Infinity,
    WindowTitle -> "Bookmark Manager",
    CellGrouping -> "Manual",
    ScrollingOptions -> {"VerticalScrollRange" -> Fit}
  ]
]



horizontalRule = 
Cell["", "Text",
    CellFrame->{{0, 0}, {0, 0.5}},
    ShowCellBracket -> False,
    CellMargins->{{0, 0}, {1, 1}},
    CellElementSpacings->{"CellMinHeight"->1},
    CellFrameMargins->False,
    CellFrameColor->GrayLevel[0.8],
    CellSize->{Inherited, 3}
]


openCopyCell[] :=
Cell[TextData[{
    ButtonBox[StyleBox["Open a Copy", FontColor -> $ControlColor],
      ButtonStyle -> "Hyperlink",
      ButtonEvaluator -> Automatic,
      ButtonFunction :> (
        Needs["AuthorTools`Bookmarks`"];
        Symbol["AuthorTools`Bookmarks`ShowBookmarks"][];
        )
    ]
  }],
  "Text",
  TextAlignment -> Right
]


newBookmarkCell[] :=
  Cell[BoxData[FormBox[addBookmarkButton[], TextForm]], "Text"]





$ControlColor = Purple;



editBookmarkButton[_[___, _[Editable, False], ___], ___] := "\[FilledSmallCircle]"

editBookmarkButton[b_Bookmark, cats_] :=
TooltipBox[ButtonBox[StyleBox["\[FilledSmallCircle]", FontColor -> $ControlColor],
  ButtonStyle -> "Hyperlink",
  ButtonFrame -> None,
  ButtonNote -> None,
  ButtonEvaluator -> Automatic,
  ButtonSource -> Cell,
  ButtonFunction :> (CompoundExpression[
    Needs["AuthorTools`Bookmarks`"],
    Symbol["AuthorTools`Bookmarks`Private`EditBookmark"][ButtonNotebook[], #1, b, cats]]&)
  ],
  tooltipCell["Edit bookmark"]
]


removeBookmarkButton[_[___, _[Deletable, False], ___], ___] := {}

removeBookmarkButton[b_Bookmark, cats_] :=
TooltipBox[ButtonBox[StyleBox["\[Times]", FontColor -> $ControlColor],
  ButtonStyle -> "Hyperlink",
  ButtonFrame -> None,
  ButtonNote -> None,
  ButtonEvaluator -> Automatic,
  ButtonFunction :> (
    Needs["AuthorTools`Bookmarks`"];
    Symbol["AuthorTools`Bookmarks`RemoveBookmark"][b, cats];
    SelectionMove[ButtonNotebook[], All, ButtonCell];
    NotebookDelete[ButtonNotebook[]])
  ],
  tooltipCell["Remove bookmark"]
]



addBookmarkButton[b_Bookmark, cats_] :=
ButtonBox[StyleBox["+", FontColor -> $ControlColor],
  ButtonStyle -> "Hyperlink",
  ButtonEvaluator -> Automatic,
  ButtonNote -> "Bookmark this page",
  ButtonFunction :> (
    Needs["AuthorTools`Bookmarks`"];
    Symbol["AuthorTools`Bookmarks`AddBookmark"][b, cats])
]

addBookmarkButton[] :=
ButtonBox["New Bookmark",
  ButtonFrame -> "DialogBox",
  ButtonEvaluator -> Automatic,
  ButtonFunction :> (
    Needs["AuthorTools`Bookmarks`"];
    Symbol["AuthorTools`Bookmarks`AddBookmark"][ButtonNotebook[]])
]






recursiveFormatBookmarks[lis_List, cats_] := recursiveFormatBookmarks[#, cats]& /@ lis

recursiveFormatBookmarks[b_Bookmark, {cats___}]:= bookmarkCell[b, {cats}]

recursiveFormatBookmarks[b:BookmarkCategory[cat_String, lis_List], {cats___}] :=
Cell @ CellGroupData[Flatten @ {
  categoryCell[b, {cats}],
  recursiveFormatBookmarks[lis, {cats,cat}]
}, Closed]






categoryCell[b:BookmarkCategory[name_, lis_], {cats___}] :=
Cell[name, "Text",
  CellMargins -> {{30 + Length[{cats}] * 15, 0}, {0, 7}},
  ShowGroupOpenCloseIcon ->  True,
  FontWeight -> "Bold"
]

bookmarkCell[b_Bookmark, {cats___}]:=
Cell[BoxData[FormBox[RowBox[Flatten @ {
    editBookmarkButton[b, {cats}],
    "  ",    
    bookmarkToButton[b],
    "  ",
    removeBookmarkButton[b, {cats}]
  }], TextForm]],
  "Text",
  CellMargins -> {{30 + Length[{cats}] * 15, 0}, {0, 0}},
  CellTags -> categoryListToString[{cats}],
  System`ShowSyntaxStyles -> False,
  PageWidth -> Infinity,
  TooltipBoxOptions -> {ActionDelay -> $tooltipDelay}
]

$tooltipDelay = 0.5;

bookmarkCell[Bookmark["Delimiter", ___?OptionQ], {cats___}]:=
Cell[BoxData[FormBox["", TextForm]],
  "Text",
  CellMargins -> {{30 + Length[{cats}] * 15, 50}, {0, 0}},
  System`ShowSyntaxStyles -> False,
  Selectable->False,
  PageWidth -> Infinity
]








tooltipCell[contents_] := 
  Cell[contents, "Text", CellMargins -> {{150, 150}, {Inherited, Inherited}}]


bookmarkToButton[b:Bookmark[name_, "Web", address_, ___]]:=
  TooltipBox[ButtonBox[Cell[name, FontColor -> GrayLevel[0]], ButtonStyle -> "Hyperlink", ButtonData :> {URL[address], None}, ButtonFrame->None, ButtonNote -> None], bookmarkToURL[b]//tooltipCell]

bookmarkToButton[b:Bookmark[name_, "Help", {cat_, tag_}, ___]] :=
  TooltipBox[ButtonBox[Cell[name], ButtonStyle -> categoryToButtonStyle[cat], ButtonData -> {tag}, ButtonNote -> None], bookmarkToURL[b]//tooltipCell]

bookmarkToButton[b:Bookmark[name_, "Help", {cat_, tag_, tag2_}, ___]] :=
  TooltipBox[ButtonBox[Cell[name], ButtonStyle -> categoryToButtonStyle[cat], ButtonData -> {tag, tag2}, ButtonNote -> None], bookmarkToURL[b]//tooltipCell]

bookmarkToButton[b:Bookmark[name_, "File", path:(_String | _FrontEnd`FileName), ___]] :=
  TooltipBox[ButtonBox[Cell[name, FontColor -> RGBColor[0,0,1]], ButtonStyle -> "Hyperlink", ButtonData -> {path, None}, ButtonFrame->None, ButtonNote -> None], bookmarkToURL[b]//tooltipCell]

bookmarkToButton[b:Bookmark[name_, "File", {path:(_String | _FrontEnd`FileName), tag_String}, ___]] :=
  TooltipBox[ButtonBox[Cell[name, FontColor -> RGBColor[0,0,1]], ButtonStyle -> "Hyperlink", ButtonData -> {path, tag}, ButtonFrame->None, ButtonNote -> None], bookmarkToURL[b]//tooltipCell]



bookmarkToURL[Bookmark[_, "Web", u_String, ___]] := u;

bookmarkToURL[Bookmark[_, "Help", {c_String, t_String}, ___]] := "HelpBrowser / " <> c <> " / " <> t

bookmarkToURL[Bookmark[_, "Help", {c_String, t_String, u_String}, ___]] := "Help Browser / " <> c <> " / " <> t <> "#" <> u

bookmarkToURL[Bookmark[_, "File", u_String | {u_String, ___}, ___]] := u

bookmarkToURL[Bookmark[_, "File", u_FrontEnd`FileName | {u_FrontEnd`FileName, ___}, ___]] := ToFileName[u]


categoryToButtonStyle[s : ("AddOns" | "RefGuide")] := s <> "LinkText";

categoryToButtonStyle[s_String] := s <> "Link"






HelpBrowserNotebookQ[nb_NotebookObject] :=
  MatchQ[NotebookInformation[nb], {___, "HelpPanel" -> _, ___}]

HelpBrowserNotebookQ[___] := False


(*
   BookmarkCurrentPage[] gets the foremost Help Browser if it happens
   to be the input notebook, which is likely. If InputNotebook[] is
   not a Help Browser, then it bookmarks the current page in
   HelpBrowserNotebook[].
*)


BookmarkCurrentPage[] :=
Block[{nb},
  Which[
    HelpBrowserNotebookQ[nb = InputNotebook[]], BookmarkCurrentPage[nb],
    HelpBrowserNotebookQ[nb = HelpBrowserNotebook[]], BookmarkCurrentPage[nb],
    True, $Failed
  ]
]

BookmarkCurrentPage[nb_NotebookObject] :=
Block[{info, b},
  info = "HelpBrowserLocation" /. NotebookInformation[nb];
  If[!MatchQ[info, {_String, s_String, ___} /; s =!= ""], Return[]];
  b = Bookmark[#2, "Help", If[Length[{##}] < 3 || #3 === "", {#1, #2}, {#1, #2, #3}]]& @@ info;
  
  AddBookmark[b, {""}];
];



BookmarkCurrentPageCell[] :=
  Cell[BoxData[FormBox[ BookmarkCurrentPageButton[], TextForm]], "Text"]


BookmarkCurrentPageButton[] := 
ButtonBox["Bookmark Current Page",
  ButtonFunction :> (CompoundExpression[
    Needs["AuthorTools`Bookmarks`"],
    Symbol["FrontEnd`MessagesToConsole"][
      Symbol["AuthorTools`Bookmarks`Private`BookmarkCurrentPageButtonFunction"][]]
    ]&), 
  ButtonEvaluator -> Automatic,
  ButtonFrame -> "DialogBox"
]


BookmarkCurrentPageButtonFunction[___] :=
Block[{},
  BookmarkCurrentPage[];
  ShowBookmarks[]
]







catToString["RefGuide"] = "Built-in Functions";
catToString["AddOns"] = "Add-ons & Links";
catToString["MainBook"] = "The Mathematica Book";
catToString["OtherInformation"] = "Front End";
catToString["GettingStarted"] = "Getting Started";
catToString["Tour"] = "Tour";
catToString["Demos"] = "Demos";
catToString["MasterIndex"] = "Master Index";






$DefaultBookmarkDataFile = ToFileName[{DirectoryName[System`Private`$InputFileName],
  "FrontEnd", "TextResources"}, "DefaultBookmarkData.m"];







ResetBookmarks::del = "Could not delete the directory: `1`"

ResetBookmarks[] := 
Block[{d = $BookmarkDirectory},
  If[FileType[d] === Directory,
    DeleteDirectory[d, DeleteContents -> True];
    If[FileType[d] === Directory,
      Message[ResetBookmarks::del, d];
      Return[$Failed]
    ]
  ];
  
  InitializeBookmarks[]
]



InitializeBookmarks::err = "Bookmarks could not be initialized.";

InitializeBookmarks[] :=
Block[{d = $BookmarkDirectory},
    
  If[FileType[d] =!= Directory, CreateDirectory[d]];
  If[FileType[d] =!= Directory, Message[InitializeBookmarks::err]; Return[$Failed]];
  
  $DefaultBookmarks = Get[$DefaultBookmarkDataFile];
  If[!MemberQ[{List, BookmarkCategory}, Head[$DefaultBookmarks]], Message[InitializeBookmarks::err]; Return[$Failed]];
  
  (* Hold off on this just yet... *)
  (* $DefaultBookmarks = setBookmarkOptions[$DefaultBookmarks, Editable -> False, Deletable -> False]; *)
  
  If[FileType[$BookmarkDataFile] =!= File, WriteBookmarkData[$BookmarkDataFile, $DefaultBookmarks]];
  If[FileType[$BookmarkNotebookFile] =!= File, WriteBookmarkNotebook[$BookmarkNotebookFile, $DefaultBookmarks]];
  
  If[Map[FileType, {$BookmarkDataFile, $BookmarkNotebookFile}] =!= {File, File},
    Message[InitializeBookmarks::err];
    Return[$Failed]
  ];  
  
  $BookmarkDirectory
]



setBookmarkOptions[x_List, opts___] := setBookmarkOptions[#, opts]& /@ x

setBookmarkOptions[BookmarkCategory[x_, y_List], opts___] := BookmarkCategory[x, setBookmarkOptions[y, opts]]

setBookmarkOptions[Bookmark[x___], opts___] := Bookmark[x, opts]




ImportBookmarks[file_String] := ImportBookmarks[Get[file]] /; FileType[file] === File


ImportBookmarks[x:(_List | _BookmarkCategory | _Bookmark)] :=
Block[{lis},
  lis = Flatten[{GetBookmarkData[], x}];
  WriteBookmarkData[$BookmarkDataFile, lis];
  WriteBookmarkNotebook[lis];
]










CategoryList[] := Flatten[{"", rGetCategoryList[GetBookmarkData[], {}] }]

rGetCategoryList[BookmarkCategory[s_String, t_List], {cats___}] :=
  {categoryListToString[{cats, s}], rGetCategoryList[t, {cats, s}]}

rGetCategoryList[lis_List, cat_] := Map[rGetCategoryList[#, cat]&, lis]

rGetCategoryList[lis_Bookmark, cat_] := {}









EditBookmark[nb_, originalCell_, b_Bookmark, cats_] :=
Block[{catString = categoryListToString[cats]},
  SelectionMove[nb, All, ButtonCell];
  NotebookWrite[nb, editingCell[originalCell, b, catString]]
]


AddBookmark[nb_] := 
Block[{},
  SelectionMove[nb, Before, ButtonCell];
  NotebookWrite[nb, editingCell[{}, Bookmark["", "Web", ""], ""]];
]
  
  





categoryListToString[{cats___String}] := StringJoin[BoxForm`Intercalate[{cats}, "/"]];

categoryStringToList[""] := {""};
categoryStringToList[cats_String] := StringSplit[cats, "/"];






editingCell[originalCell_, b:Bookmark[name_, location_, u_, opts___], catString_] :=
Module[{url = u},

  If[Head[url] =!= String, url = ToString[url, InputForm]];
  
  Cell[BoxData[PanelBox[RowBox[{GridBox[{
      { "Name:", InputFieldBox[name, String]},
      { "", PopupMenuBox[location, {
        "File" -> StyleBox["Local File", Small],
        "Help" -> StyleBox["HelpBrowserLocation", Small],
        "Web" -> StyleBox["Web URL", Small]}]},
      { "Address:", InputFieldBox[url, String]},
      { "Category:", InputFieldBox[catString, String]}},
      ColumnAlignments -> {Right,Left},
      RowSpacings -> 0.5],
      "\n",
      cancelEditButton[originalCell],
      "  ",
      doneEditButton[b, catString]}]]],
    "Text",
    CellMargins -> {{10,10},{7,7}},
    ButtonBoxOptions -> {ButtonFrame -> "DialogBox"},
    System`ShowSyntaxStyles->False,
    AutoItalicWords->{},
    AutoSpacing -> False,
    AutoIndent -> False,
    PageWidth->Infinity,
    ShowStringCharacters -> True
  ]
]


cancelEditButton[originalCell_Cell] :=
ButtonBox["Cancel",
  Active->True,
  ButtonEvaluator -> None,
  ButtonFunction :> (FrontEndExecute[{
    SelectionMove[ButtonNotebook[], All, ButtonCell],
    NotebookWrite[ButtonNotebook[], originalCell] }]&)
]

cancelEditButton[{}] :=
ButtonBox["Cancel",
  Active->True,
  ButtonEvaluator -> None,
  ButtonFunction :> (FrontEndExecute[{
    SelectionMove[ButtonNotebook[], All, ButtonCell],
    NotebookDelete[ButtonNotebook[]] }]&)
]




doneEditButton[args___] := 
ButtonBox["OK",
  Active->False,
  Enabled -> False,
  ButtonSource -> Cell,
  ButtonEvaluator -> Automatic,
  ButtonFunction :> (doneEditButtonFunction[#1, ButtonNotebook[], args]&)
]


doneEditButtonFunction[newCell_Cell, nb_, b_Bookmark, c_] :=
Block[{name, loc, category, url, new, cats},
  new = Cases[newCell, (InputFieldBox | PopupMenuBox)[x_String,___] :> x, Infinity];
  If[Length[new] =!= 4, Print[{"Not enough strings", new}]; Return[$Failed]];
  {name, loc, url, category} = new;
  If[loc === "Help", url = ToExpression[url]];
  
  new = Bookmark[name, loc, url];
  cats = categoryStringToList @ category;
  
  Which[
    (* if nothing changed, don't do anything *)
    c === category && b === new,
    SelectionMove[nb, All, ButtonCell];
    NotebookWrite[nb, bookmarkCell[new, cats]];
    ,
    (* if the category is the same, we can just replace the cell in place *)
    c === category,
    SelectionMove[nb, All, ButtonCell];
    NotebookWrite[nb, bookmarkCell[new, cats]];
    WriteBookmarkData[$BookmarkDataFile, GetBookmarkData[] /. b -> new];
    ,
    (* otherwise, refresh the whole listing *)
    True,
    RemoveBookmark[b, cats];
    AddBookmark[new, cats];
    ShowBookmarksInNotebook[nb]
  ];
  
  new
]







(* *********** WriteBookmarkData **************** *)

(* paraphrased from AuthorTools`MakeCategories`WriteBrowserCategories *)


blanks[n_] := blanks[n] = StringJoin[Table[" ", {n}]];

recursiveWriteBC[st_, {}] := WriteString[st, "{}"]

recursiveWriteBC[st_, lis_List] :=
  (
    WriteString[st, "{\n"];
    (recursiveWriteBC[st, #, 2]; WriteString[st, ",\n"])& /@ Most[lis];
    recursiveWriteBC[st, Last[lis], 2];
    WriteString[st, "\n}\n"];
  );


recursiveWriteBC[st_, BookmarkCategory[str_, {}], d_] :=
  (
    WriteString[st, blanks[d]];
    WriteString[st, "BookmarkCategory[", toStr[str], ", "];
    WriteString[st, "{ }]"];
  );

recursiveWriteBC[st_, BookmarkCategory[str_, lis_List], d_] :=
  (
    WriteString[st, blanks[d]];
    WriteString[st, "BookmarkCategory[", toStr[str], ", "];
    WriteString[st, "\n", blanks[d+2], "{\n"];
    (recursiveWriteBC[st, #, d+4]; WriteString[st, ",\n"])& /@ Most[lis];
    recursiveWriteBC[st, Last[lis], d+4];
    WriteString[st, "\n", blanks[d+2], "}\n", blanks[d], "]"];
  );

recursiveWriteBC[st_, other_, d_] :=
    WriteString[st, blanks[d], toStr[other]];


WriteBookmarkData[path_, lis_List] :=
Block[{st},
  st = OpenWrite[path, PageWidth -> Infinity];
  recursiveWriteBC[st, lis];
  Close[st];
  path
]

quoteIfString[x_String] := "\"" <> x <> "\"";
quoteIfString[x_] := x;
toStr[x_] := ToString[x, InputForm, CharacterEncoding -> "ASCII"]










(* Install the default bookmarks if there are no bookmarks present. *)

InitializeBookmarks[]












End[]

EndPackage[]
