(* :Context: AuthorTools`Search` *)

(* :Author: Louis J. D'Andria *)

(* :Summary:
    This package defines code used for searching the 
    help browser, notebooks, web sites, etc.
*)

(* :Copyright: *)

(* :Package Version: $Revision: 1.33 $ $Date: 2004/09/22 16:17:02 $ *)

(* :Mathematica Version: 6.0 *)

(* :History:

*)

(* :Keywords:
     
*)

(* :Discussion:
    
*)

(* :Warning:
    
*)





BeginPackage["AuthorTools`Search`",
  {"AuthorTools`Common`", "AuthorTools`Experimental`", "WebServices`"}]



HelpBrowserSearch

ExpressionHelp

(*
WolframWebSearch
ItemNameSearch
AddOnsSearch
*)

IncludeWebSearch
IncludeItemNameSearch
IncludeAddOnsSearch
IncludeExcerpts

SearchLimit
SearchSites
$SearchSites
SearchLanguages


InitializeSearch




Begin["`Private`"]




startProgress[___] := 
Block[{x},
  If[ (x = HelpBrowserNotebook[]; Head[x] === NotebookObject) &&
      (x = NotebookInformation[x]; ListQ @ x) &&
      (x = "HelpPanel" /. x; Head[x] === NotebookObject),
    $panelnb = x,
    $panelnb = False
  ]
]

showProgress[x_] :=
If[$panelnb === False,
  CellPrint[Cell[x, "Print"]]
  ,
  SelectionMove[$panelnb, After, Notebook];
  NotebookWrite[$panelnb, Cell[x, "Text", FontSlant -> "Italic", CellTags -> "progress"]];
]

endProgress[___] :=
If[$panelnb === False,
  Null
  ,
  NotebookFind[$panelnb, "progress", All, CellTags, AutoScroll->False];
  NotebookDelete[$panelnb]
]




$SearchServiceURL = "http://webservices.devel.wolfram.com/services/SearchServices/WolframSearch.m?wsdl";


networkCheck[] := Block[{mess, res},
  mess = Head[XML`Parser`XMLGet::"prserr"];
  Off[XML`Parser`XMLGet::"prserr"];
  res = $Failed =!= 
    XML`Parser`XMLGet[$SearchServiceURL];
  If[mess =!= $Off, On[XML`Parser`XMLGet::"prserr"]];
  res
];



InitializeSearch[] :=
Block[{},
(*
Reading the help browser confiruation files takes 10 seconds or so on
the first launch. This will be replaced by a routine to get this
information directly from the front end.
*)
  
  startProgress[];
  showProgress["Reading Browser Configuration Files\[Ellipsis]"];
  GetBrowserLookupTable[];

(*
Starting the search service takes 10 seconds or so per kernel launch.
The first search will also take a few extra seconds to finish setting
up the web services stuff. If there is no network available, then the
$NetworkQ definition should time out after a few seconds.
*)

  If[$EnableNetworkedSearch === False,
    showProgress["Network Use Disabled."];
    $NetworkQ = False
    ,
    showProgress["Testing Network\[Ellipsis]"];
    $NetworkQ = networkCheck[]
  ];
  
  If[$NetworkQ,
    showProgress["Starting Search Service\[Ellipsis]"];
    $SearchServiceSetup = InstallService[$SearchServiceURL, "WebServices`"]
    ,
    showProgress["Search Service Unavailable."]
  ];
  
  showProgress["Finishing Setup\[Ellipsis]"];
  (* endProgress[]; *)

]



InitializeSearch[]










wolf[n_] := 
 Graphics[Text[
   System`RotationBox[StyleBox["\[Wolf]", "Title"], 
    System`BoxRotation -> n], {0, 0}, {0, 0}, {-1, 0}], ImageSize -> {50, 50}]

spinningWolf = 
 RowBox[{DynamicBox[
    ToBoxes[Global`wolf[FrontEnd`Private`localvar$0], StandardForm]], 
   AnimatorBox[FrontEnd`Private`localvar$0, {0, 2 Pi - Pi/4, Pi/4}, 
    DefaultDuration -> 1]}];






$SearchLimit = 20;


ItemNameSearch[str_] :=
Module[{lis, t},
  t = Timing[lis = ItemLookup[str, True, True, ItemLookupCategories -> $HelpCategories];];
  ShowSearchResults[str, "Item Names", Length[lis], First[t], lis, {}]
  ]


formatMatch[str_, {{"Help Browser", cat_, cats___, item_}, tag_}] :=
  locationButton[str, {{"Help Browser", cat, cats, item}, tag}, True]


locationButton[str_, {{"Help Browser", cat_, cats___, item_}, tag_}, bullet_, opts___] :=
{If[TrueQ @ bullet, "\[FilledSmallCircle]\[NonBreakingSpace]", {}],
TooltipBox[
  ButtonBox[Cell[highlightdata[cat <> " / " <> item, str], opts],
    ButtonStyle -> categoryToButtonStyle[cat],
    ButtonData -> tag,
    ButtonNote -> None],
  tooltipCell[
    StringJoin[BoxForm`Intercalate[{"Help Browser", cat, cats, tag}, " / "]]]
]} // Flatten // RowBox

categoryToButtonStyle[cat_] := Block[{sty},
  sty = AuthorTools`Experimental`Private`categoryToButtonStyle[cat];
  If[MemberQ[{"RefGuideLink", "AddOnsLink"}, sty], sty <> "Text", sty]
  ]

tooltipCell[contents_] := 
  Cell[contents, "Text", CellMargins -> {{150, 150}, {Inherited, Inherited}}]









WolframWebSearch[___] := 
 CellPrint[
  resultCell[
   Cell["Web searches require a network connection.", FontWeight -> "Bold", 
    FontColor -> Purple]]] /; Not @ $NetworkQ


WolframWebSearch[str_, opts___?OptionQ] := WolframWebSearch[str, "Documents_v6", opts]

WolframWebSearch[str_, restrict_, opts___?OptionQ] := WolframWebSearch[str, restrict, False, opts]

WolframWebSearch[str_, restrict_, excerpts_, opts___?OptionQ] :=
Module[{total, timing, matches, sugg},
  vPrint[excerpts];
  lis = WolframSearch[{
    "Query" -> str,
    "Restrict" -> restrict,
    "Limit" -> ToString[$SearchLimit],
    "ReturnTypes" -> Flatten @ 
      {"Title", "Url", "HelpBrowserLocation", If[TrueQ[excerpts], "Excerpt", {}]},
    "Filter" -> False,
    opts}
  ];

  {total, timing, matches, sugg} = {"TotalMatches", "SearchTime", "Matches", "Suggestion"} /. lis;
    
  (* WolframSearch returns -1 when there are no matches. *)
  If[total < 0, total = 0];
  If[matches === Null, matches = {}];
  If[sugg === "Suggestion", sugg = {}, sugg = {sugg}];
  sugg = Join[sugg, tokensInString[str]];

  ShowSearchResults[str, (restrict /. restrictstable), total, timing, matches, sugg]
]




tokensInString[str_String] :=
  Intersection[StringSplit @ str, First /@ tokenTranslationRules] /. tokenTranslationRules


tokenTranslationRules =
{
  "<<>>" -> "skeleton",
  "<>" -> "StringJoin",
  "<<" -> "Get",
  ">>" -> "Put",
  ">>>" -> "PutAppend",
  "<=" -> "LessEqual",
  "<" -> "Less",
  ">=" -> "GreaterEqual",
  ">" -> "Greater",
  "//@" -> "MapAll",
  "/@" -> "Map",
  "@@@" -> "Apply",
  "@@" -> "Apply",
  "@" -> "function application",
  "@" -> "string metacharacter",
  "=!=" -> "UnsameQ",
  "!=" -> "Unequal",
  "===" -> "SameQ",
  "==" -> "Equal",
  "^=" -> "UpSet",
  "^:=" -> "UpSetDelayed",
  "/::=" -> "TagSetDelayed",
  "/:=." -> "TagUnset",
  "/:" -> "TagSet",
  ":=" -> "SetDelayed",
  "=." -> "Unset",
  "##" -> "SlotSequence",
  "#" -> "Slot",
  "%%" -> "Out",
  "%" -> "Out",
  "^^" -> "number base"
  "^" -> "Power",
  "!!" -> "Factorial2",
  "!!" -> "show file",
  "!" -> "Factorial",
  "!" -> "pipe prefix",
  "!" -> "shell escape",
  "**" -> "NonCommutativeMultiply",
  "*=" -> "TimesBy",
  "/=" -> "DivideBy",
  "(*" -> "comment",
  "*" -> "Times",
  "*" -> "string metacharacter",
  
  "[[]]" -> "Part",
  "[[" -> "Part",
  "[]" -> "function application",
  "[" -> "function application",
  "{}" -> "List",
  "{" -> "List",
  "||" -> "Or",
  "|" -> "Alternatives",
  "&&" -> "And",
  "&" -> "Function",
  "\\\\" -> "line continuation",
  
  "//." -> "ReplaceRepeated",
  "/." -> "ReplaceAll",
  "\\." -> "special characters",
  "\\:" -> "special characters",
    
  "___" -> "BlankNulSequence",
  "__" -> "BlankSequence",
  "_." -> "Optional",
  "_:" -> "Optional",
  "_" -> "Blank",
  "..." -> "RepeatedNull",
  ".." -> "Repeated",
  "." -> "Dot",
  "/;" -> "Condition",
  ";" -> "CompoundExpression",
  "??" -> "Information",
  "?" -> "PatternTest",
  "?" -> "Information",
  ":" -> "Pattern",

  "++" -> "Increment",
  "--" -> "Decrement",
  "+=" -> "AddTo",
  "-=" -> "SubtractFrom",
  "->" -> "Rule",
  ":>" -> "RuleDelayed",
  
  "-" -> "Subtract",
  "+" -> "Plus",
  "/" -> "Divide",
  "~" -> "function application",
  "~" -> "home directory",
  
  "//" -> "function application",
  "'" -> "Derivative",
  "``" -> "StringForm",
  "``" -> "accuracy mark",
  "`" -> "context mark",
  "`" -> "StringForm",
  "`" -> "number mark",
  "`" -> "precision mark"
};






$ErrorLogFile = ToFileName[{DirectoryName[$BrowserCacheFile]}, "SearchErrorLog.m"];

$LogSearchErrors = True; (* turn off before we ship *)

logError[args___] := Null /; !$LogSearchErrors

logError[args___] := Block[{st},
  st = OpenAppend[$ErrorLogFile, PageWidth -> Infinity];
  Write[st, args];
  Close[st]]




formatMatch[str_, opts : {___Rule, "Excerpt" -> _, ___Rule}] := formatWithExcerpt[str, opts]

formatMatch[str_, opts : {__Rule}] := locationButton[str, opts]


locationButton[str_, opts : {__Rule}] :=
Block[{title, url, help, cat, tag, tag2, match},
  {title, url, help} = {"Title", "Url", "HelpBrowserLocation"} /. opts;
  title = highlightdata[htmltostring[title], str];
  {
   webLink[Cell["\[FilledSmallCircle]", FontColor -> GrayLevel[0]], url],
   "\[NonBreakingSpace]",  
   If[help === "HelpBrowserLocation",
     webLink[Cell[title, FontColor -> GrayLevel[0]], url],
     {cat, tag, tag2} = ToExpression[help];
     match = matchingItem[{cat, tag, tag2}];
     If[match === {},
      logError[{"No matching item", help, url}];
      webLink[Cell[title, FontColor -> GrayLevel[0.4]], url],
      locationButton[str, match, False]
      ]
   ]
  } // Flatten // RowBox
]


formatWithExcerpt[str_, opts___] := 
Block[{title, url, help, cat, tag, tag2, match, excerpt, fontopts},
  {title, url, help, excerpt} = {"Title", "Url", "HelpBrowserLocation", "Excerpt"} /. opts;
  title = highlightdata[htmltostring[title], str];
  excerpt = highlightdata[htmltostring[excerpt], str];
  
  fontopts = Sequence[FontFamily -> "Helvetica" (*, FontWeight -> "Bold" *)];
  {{
   webLink[Cell["\[FilledSmallCircle]", FontColor -> GrayLevel[0]], url],
   "\[NonBreakingSpace]",  
   If[help === "HelpBrowserLocation",
     webLink[Cell[title, FontColor -> GrayLevel[0], fontopts], url],
     {cat, tag, tag2} = ToExpression[help];
     match = matchingItem[{cat, tag, tag2}];
     If[match === {},
      logError[{"No matching item", help, url}];
      webLink[Cell[title, FontColor -> GrayLevel[0.4], fontopts], url],
      locationButton[str, match, False, fontopts]
      ]
   ]} // Flatten // RowBox,
   Cell[excerpt, FontFamily->"Times"]
  }
]



webLink[contents_, url_] := 
TooltipBox[
  ButtonBox[contents, ButtonStyle -> "Hyperlink", 
    ButtonData -> {URL[url], None}, ButtonFrame -> None, ButtonNote -> None],
  tooltipCell[url]
]

buttonContents[category_, title_] := 
 StringJoin[
  First @ AuthorTools`Experimental`Private`nameToCategoryList @ 
    category , " / ", title]

buttonContents2[category_, title_, url_] := 
 StringReplace[
  url, {"http://documents.devel.wolfram.com/v6/" -> "", ".html" -> ""}]



htmltostring[str_] := translateAllEntities[str]

highlightdata[content_, str_] := 
Block[{x},
  TextData @ StringReplace[
    content,
    x:(StringSplit[str, {" ","+","*","~",","}]) :> StyleBox[x, FontWeight->"Bold"],
    IgnoreCase -> True
  ] /. StringExpression -> List
]



translateAllEntities[str_] := decodeString @ StringReplace[str,
    Reverse /@ System`Convert`MLStringDataDump`$UnicodeToHTML4Entities]

decodeString[str_] := StringReplace[str, {
   "&lt;"->"<",
   "&gt;"->">",
   "&amp;"->"&",
   "&quot;"->"\"",
   "\\"->"\\[Backslash]",
   (* apos isn't translated properly *)
   "&apos;" -> "'",
   (* remaining rules are temporary, until we have real html-to-notebook support *)
   "<b>" -> "",
   "</b>" -> "",
   "<br>" -> ""
}]





matchingItem[{cat_, tag_, _}] := 
 Block[{catname, res}, 
  catname = 
   First @ AuthorTools`Experimental`Private`nameToCategoryList @ cat;
  res = Cases[$BrowserLookupTable, {_, tag, _, 
      catlist : {"Help Browser", catname, __}} :> {catlist, tag}, 1, 1];
  If[res === {}, {}, First[res]]]



AddOnsSearch[str_] :=
Module[{res},
  res = NotebookSearch[{$BaseDirectory, $UserBaseDirectory}, str, "HelpBrowserOutput",
     Verbosity -> 0];
  lis = First @ res;
  lis = {First[#], "IndexTag" /. Rest[#]} & /@ lis;
  ShowSearchResults[str, "Local AddOns", res[[4]], res[[2,1]], lis, {}]
]




SetAttributes[ExpressionHelp, HoldAllComplete]

ExpressionHelp[expr_] :=
Block[{lis},
  lis = Cases[Unevaluated[expr[]], s_Symbol :> SymbolName[Unevaluated[s]], Infinity, Heads -> True];
  lis = DeleteCases[matchingItem[{"RefGuide", #, ""}] & /@ Union[lis], {}];
  ShowSearchResults["Symbols", "the reference guide", Length[lis], 0, lis, {}]
]





(* ShowSearchResults uses CellPrint if this was called from
a function definition, or NotebookWrite if this was called in 
the side panel of the Help Browser *)


ShowSearchResults[args___] :=
  If[TrueQ @ $FromHelpPanel, WriteResultsToHelpPanel[args], PrintSearchResults[args]]




PrintSearchResults[str_, s_, total_Integer, timing_, matches_List, sugg_] :=
Module[{lis, e},
  lis = Take[matches, Min[Length[matches], $SearchLimit]];
  e = MatchQ[lis, {{___, "Excerpt" -> _, ___}, ___}];
  vPrint[lis];
  lis = formatMatch[str, #]& /@ lis;
  CellPrint @ resultCell @ GridBox[Map[List, Flatten @
    {header[str, s, Length[lis], total], lis,
     moreButton[Length[lis], total], suggestionButton[sugg]}],
     ColumnAlignments -> Left,
     RowSpacings -> If[e,
       Flatten @ Table[{0,2},{1 + Length @ lis}],
       Flatten @ {0,2, Table[0, {-1 + Length @ lis}], 2, 0}]]
]


header[str_, s_, _, 0] := {
  Cell[ToString @ StringForm["'`1`' not found in `2`", str, s], FontWeight->"Bold", FontFamily -> "Helvetica"]
}

header[str_, s_, k_Integer, n_Integer] := {
  Cell[ToString @ StringForm["'`1`' found in `2`", str, s], FontWeight -> "Bold",  FontFamily -> "Helvetica"],
  Cell[ToString @ StringForm["Results `1`-`2` of `3`", 1, k, n], FontSlant -> "Italic"]
}  


moreButton[k_, n_] := 
If[k < n,
  {Cell[ToString[n - k] <> " more\[Ellipsis]", FontSlant -> "Italic"]},
  {}
]


suggestionButton[{strs___String}] := 
If[Length[{strs}] > 0,
  {RowBox @ Flatten @ {
     Cell["See also: ", FontSlant -> "Italic"],
     If[TrueQ @ $FromHelpPanel,
       BoxForm`Intercalate[doSuggestedSearchButton /@ {strs}, ", "],
       BoxForm`Intercalate[{strs}, ", "]
     ]}
  },
  {}
]


doSuggestedSearchButton[str_String] :=
TooltipBox[ButtonBox[str,
    ButtonStyle -> "Hyperlink",
    ButtonEvaluator -> Automatic,
    ButtonFrame -> None,
    ButtonNote -> None,
    ButtonFunction :> (DoSearchInHelpPanel[ButtonNotebook[], str]&)
  ],
  tooltipCell["Search for '" <> str <> "'"]
]




resultCell[contents_] := 
 Cell[BoxData @ FormBox[contents, TextForm], "Print", FontFamily -> "Times", 
  Background -> RGBColor[0.964706, 0.929412, 0.839216], 
  CellMargins -> {{20, Inherited}, {Inherited, Inherited}},
  CellTags -> "SearchResult",
  TooltipBoxOptions -> {ActionDelay -> $tooltipDelay}]


$tooltipDelay = 0.5;

$RecentSearches = {};

$SearchSites = {"Documents_v6"};

Options[HelpBrowserSearch] = {
  SearchLimit -> 20,
  IncludeItemNameSearch -> False,
  IncludeWebSearch -> True,
  IncludeAddOnsSearch -> False,
  IncludeExcerpts -> False,
  SearchSites :> $SearchSites,
  SearchLanguages -> All
  };

HelpBrowserSearch[opts___?OptionQ] := 
 If[StringQ[#] && # =!= "", HelpBrowserSearch[#, opts], Null] & @ 
  InputString["Enter search string."]

HelpBrowserSearch[str_String, opts___?OptionQ] :=
 Block[{$SearchLimit, q, w, a, o},
  AppendTo[$RecentSearches, str];
  {q, w, a, e, s, 
    l, $SearchLimit} = {IncludeItemNameSearch, IncludeWebSearch, 
     IncludeAddOnsSearch, IncludeExcerpts, SearchSites, SearchLanguages, SearchLimit} /. 
    Flatten[{opts, Options[HelpBrowserSearch]}];
  s = Flatten[{s}]; 
  l = Flatten[{l}] //. {"English" -> "lang_en", "Japanese" -> "lang_ja", 
     "German" -> "lang_de", "Spanish" -> "lang_sp"};
  
  removePreviousSearchResults[];
  
  If[q, ItemNameSearch[str]];
  If[w, If[l === {All},
    Scan[WolframWebSearch[str, #, TrueQ[e]] &, s],
    Outer[WolframWebSearch[str, #1, TrueQ[e], "Language" -> #2] &, s, l]
    ]];
  If[a, AddOnsSearch[str]];
  ];



removePreviousSearchResults[] := 
Block[{nb = ButtonNotebook[]},
  If[$Failed =!= nb &&
     $Failed =!= NotebookFind[nb, "SearchResult", All, CellTags],
    NotebookDelete[nb]
  ];
]










NewSearchInHelpPanel[nbobj_NotebookObject, str_String:""] :=
(
  NotebookPut[HelpPanelNotebook[str], nbobj];
  NotebookFind[nbobj, "SearchButtonCell", All, CellTags, AutoScroll -> False];
  SelectionMove[nbobj, Before, CellContents];
  SelectionMove[nbobj, Next, Character, 3];
  SetSelectedNotebook[nbobj];
  nbobj
)
 


HelpPanelNotebook[str_String] :=
Notebook[{
    Cell["Search", "Section"],
    SearchButtonCell[str],
    horizontalRule
  },
  StyleDefinitions -> "HelpBrowser.nb",
  ScrollingOptions -> {"VerticalScrollRange" -> Fit},
  WindowTitle -> "Extended Search"
]


horizontalRule = 
Cell["", "Text",
    CellFrame->{{0, 0}, {0, 0.5}},
    ShowCellBracket -> False,
    CellMargins->{{0, 0}, {1, 1}},
    CellElementSpacings->{"CellMinHeight"->1},
    CellFrameMargins->False,
    CellFrameColor->GrayLevel[0.8],
    CellSize->{Inherited, 3}
]



restrictstable = {
  "Documents_v6" -> "Help Browser",
  "www" -> "Main Wolfram Site",
  "Support" -> "Technical FAQs",
  "Store" -> "Wolfram Store",
  "InfoCenter" -> "Information Center",
  "Documents" -> "Documentation Center",
  "Main" -> "All above",
  "MathGroup" -> "MathGroup",
  "Forums" -> "Forums",
  "ScienceWorld" -> "MathWorld/ScienceWorld",
  "WolframScience" -> "WolframScience.com",
  "StephenWolfram" -> "StephenWolfram.com"
}


$DefaultRestrict = "Documents_v6";

sitePopup[] :=
Block[{lis = {}, default, popuprow, popupdelimiter},
  If[$NetworkQ, lis = WebServices`GetCategories[]];
  
  lis = Intersection[First /@ restrictstable, lis];
  PrependTo[lis, "ItemNames"];
  
  default = If[MemberQ[lis, $DefaultRestrict], $DefaultRestrict, First[lis]];
  
  popuprow[r_] := If[MemberQ[lis, r], r -> (r /. restrictstable), {}];
  popupdelimiter = default -> "";
  
  PopupMenuBox[
    default,
    Flatten @ {
        "ItemNames" -> "Browser Item Names",
        popuprow["Documents_v6"],
        popupdelimiter,
        popuprow /@ {"www","Support","Store","InfoCenter","Documents","Main"},
        popupdelimiter,
        popuprow /@ {"MathGroup","Forums","ScienceWorld","WolframScience","StephenWolfram"}
    },
    FieldSize -> {{1,40},1}
  ]
]




DoSearchInHelpPanel[nb_NotebookObject, str_String] :=
(
  FrontEnd`Private`SetHelpPanelMode[nb, "Search"];
  FrontEnd`MessagesToConsole[
      SearchButtonFunction[ SearchButtonCell[str] ]
  ]
)



SearchButtonFunction[c_Cell] :=
Block[{str, site, $FromHelpPanel = True},
  str = StringJoin[Cases[c, InputFieldBox[x_, ___] :> Cases[{x}, _String, Infinity], Infinity, 1]];
  site = Cases[c, PopupMenuBox[x_, ___] :> x, Infinity];
  startProgress[];
  showProgress["Searching for " <> str <> "\[Ellipsis]"];
  Switch[site,
    {"ItemNames"}, HelpBrowserSearch[str, IncludeItemNameSearch -> True, IncludeWebSearch -> False],
    {_String}, HelpBrowserSearch[str, SearchSites -> site],
    _, HelpBrowserSearch[str]
  ];
  endProgress[];
]


SearchButtonCell[str_String] :=
Cell[BoxData @ FormBox[RowBox @ Flatten @ {
   sitePopup[],
   "\n",
   InputFieldBox[str, String, FieldSize -> {20, 1}], " ", 
   ButtonBox["Search",
    ButtonFunction:>(CompoundExpression[
      Needs["AuthorTools`Search`"],
      Symbol["FrontEnd`MessagesToConsole"][
        Symbol["AuthorTools`Search`Private`SearchButtonFunction"][#1]]
      ]&),
    ButtonEvaluator->Automatic,
    ButtonSource->Cell,
    ButtonFrame->"DialogBox"]}, TextForm],
  "Text",
  System`ShowSyntaxStyles->False,
  AutoItalicWords->{},
  AutoSpacing -> False,
  AutoIndent -> False,
  CellTags -> "SearchButtonCell",
  ShowStringCharacters -> True
]






WriteResultsToHelpPanel[str_, s_, total_Integer, timing_, matches_List, sugg_] :=
Module[{lis, cell, nb = ButtonNotebook[], e},
  lis = Take[matches, Min[Length[matches], $SearchLimit]];
  e = MatchQ[lis, {{___, "Excerpt" -> _, ___}, ___}];
  vPrint[lis];
  lis = formatMatch[str, #]& /@ lis;
  cell = resultCell @ GridBox[Map[List, Flatten @
    {header[str, s, Length[lis], total], lis,
     moreButton[Length[lis], total], suggestionButton[sugg]}],
     ColumnAlignments -> Left,
     RowSpacings -> If[e,
       Flatten @ Table[{0,2},{1 + Length @ lis}],
       Flatten @ {0,2, Table[0, {-1 + Length @ lis}], 2, 0}]];
  
  cell[[2]] = NotebookDefault; (* Using an output style like "Print" causes problems *)
  
  SelectionMove[nb, After, Notebook, AutoScroll -> False];
  NotebookWrite[nb, cell, After, AutoScroll -> False];
]








End[]

EndPackage[]
