(* ::Package:: *)

(* :Title: Atomic Data for Resonance Absorption Lines *)

(* :Author: Barbara Ercolano *)

(* :Summary:
This package provides functions for a fast and efficient search through the 
atomic data table for resonance absorption lines.  It also provides 
functions for drawing absorption maps of particular elements or of a 
particular wavelength range.
*)

(* :Package Version: 1.1 *)

(* :Copyright: Copyright 1997-2007, Wolfram Research, Inc. *)

(* :Context: ResonanceAbsorptionLines` *)

(* :History:
	Version 1.0 by Barbara Ercolano (Wolfram Research), 1997.
	Version 1.1 by ECM, usage message revisions, 1998.
    Edited to fit the new paclet system, 2006.
*)

(* :Keywords: *)

(* :Source: *)

(* :Mathematica Version: 3.0-6.0 *)

BeginPackage["ResonanceAbsorptionLines`", "Units`"];

If[Not@ValueQ[Element::usage],Element::usage="Element is the element producing resonance absorption lines."];

If[Not@ValueQ[IonStage::usage],IonStage::usage=
  "IonStage is the ionization level of the element producing resonance \
absorption lines."];

If[Not@ValueQ[Deuterium::usage],Deuterium::usage=   
   "Deuterium is an isotope of Hydrogen."];

If[Not@ValueQ[FindIons::usage],FindIons::usage=
  "FindIons[wavelength1, wavelength2] gives the resonance absorption lines in \
the wavelength range from wavelength1 to wavelength2."];

If[Not@ValueQ[AtomicData::usage],AtomicData::usage=
"AtomicData[element] gives the spectral data of the resonance absorption lines \
produced by the specified element. AtomicData[element, ionstage] gives the \
data for the lines of the ionization level ionstage."];

If[Not@ValueQ[VacuumWavelength::usage],VacuumWavelength::usage=
"VacuumWavelength[element] gives the the wavelength in vacuum of the \
resonance absorption lines produced by the specified element. \
VacuumWavelength[element, ionstage] gives the wavelength in vacuum for the lines \
of the ionization level ionstage."];

If[Not@ValueQ[AirWavelength::usage],AirWavelength::usage=
"AirWavelength[element] gives a list of pairs {wv, wa}, \
where wv is the wavelength in vacuum and wa is the wavelength in air, of the \
resonance absorption lines produced by the specified element. \
AirWavelength[element, ionstage] gives a list of pairs {wv, wa} for the  \
lines of the ionization level ionstage."];

If[Not@ValueQ[TransitionProbability::usage],TransitionProbability::usage=
"TransitionProbability[element] gives a list of pairs {wv, prob}, \
where wv is the wavelength in vacuum and prob is the spontaneous transition \
probability of the resonance absorption lines produced by the specified  \
element. TransitionProbability[element, ionstage] gives a list of pairs \
{wv, prob} for the lines of the ionization level ionstage."];

If[Not@ValueQ[LowerTermFineStructureEnergy::usage],LowerTermFineStructureEnergy::usage=
"LowerTermFineStructureEnergy[element] gives a list of pairs {wv, energy}, \
where wv is the wavelength in vacuum and energy is the energy of the \
fine-structure level in the lower term of the resonance absorption lines \
produced by the specified element. LowerTermFineStructureEnergy[element, \
ionstage] gives a list of pairs {wv, energy} for the lines of the \
ionization level ionstage."];

If[Not@ValueQ[LowerStatisticalWeight::usage],LowerStatisticalWeight::usage=
"LowerStatisticalWeight[element] gives a list of pairs {wv, weight}, \
where wv is the wavelength in vacuum and weight is the statistical weight \
of the lower level of the resonance absorption lines produced by the \
specified element. LowerStatisticalWeight[element, ionstage] gives a list of \
pairs {wv, weight} for the lines of the ionization level ionstage."];

If[Not@ValueQ[UpperStatisticalWeight::usage],UpperStatisticalWeight::usage=
"UpperStatisticalWeight[element] gives a list of pairs {wv, weight}, \
where wv is the wavelength in vacuum and weight is the statistical weight \
of the upper term of the resonance absorption lines produced by the \
specified element. UpperStatisticalWeight[element, ionstage] gives a list of \
pairs {wv, weight} for the lines of the ionization level ionstage."]; 

If[Not@ValueQ[RelativeStrength::usage],RelativeStrength::usage=
"RelativeStrength[element] gives a list of pairs {wv, strength}, \
where wv is the wavelength in vacuum and strength is the relative strength of \
the resonance absorption lines in the multiplets produced by a given element. \
RelativeStrength[element, ionstage] gives a list of pairs {wv, strength} for \
the lines of the ionization level ionstage."];

If[Not@ValueQ[DampingConstant::usage],DampingConstant::usage=
"DampingConstant[element] gives a list of pairs {wv, damping}, \
where wv is the wavelength in vacuum and damping is the natural damping \
constant of the resonance absorption lines produced by the specified element. \
DampingConstant[element, ionstage] gives a list of pairs {wv, damping} for \
the lines of the ionization level ionstage."];

If[Not@ValueQ[OscillatorStrength::usage],OscillatorStrength::usage=
"OscillatorStrength[element] gives a list of pairs {wv, strength}, \
where wv is the wavelength in vacuum and strength is the oscillator \
strength of the resonance absorption lines produced by the specified element. \
OscillatorStrength[element, ionstage] gives a list of pairs {wv, strength} for \
the lines of the ionization level ionstage."];

If[Not@ValueQ[WavelengthAbsorptionMap::usage],WavelengthAbsorptionMap::usage=
"WavelengthAbsorptionMap[wav1, wav2] gives the absorption map in the range \
specified by wav1 and wav2."];

If[Not@ValueQ[ElementAbsorptionMap::usage],ElementAbsorptionMap::usage=
"ElementAbsorptionMap[element] gives the absorption map of the specified \
element. ElementAbsorptionMap[element, ionstage] gives the \
absorption map of the element at the specified ionization level."]; 



Begin["`Private`"]


prop = {{"H", "I", "1", "2S -  2p  2Po", Null, 1215.6701, 0., 2, 6, 1., 
    6.2659*^8, 0., 0.41642}, 
   {"H", "I", "1", "2S -  2p  2Po", Null, 1215.6737, 0., 2, 2, 0.33333, 
    6.2649*^8, 6.2649*^8, 0.13889}, 
   {"H", "I", "1", "2S -  2p  2Po", Null, 1215.6683, 0., 2, 4, 
    0.66667, 6.2651*^8, 6.2651*^8, 0.27761}, 
   {"H", "I", "2", "2S -  3p  2Po", Null, 1025.7223, 0., 2, 6, 1., 
    1.6729*^8, 2.245*^7, 0.079118}, 
   {"H", "I", "2", "2S -  3p  2Po", Null, 1025.723, 0., 2, 2, 
    0.33333, 1.6729*^8, 1.8965*^8, 0.026372}\
, {"H", "I", "2", 
      "2S -  3p  2Po", Null, 1025.7219, 0., 2, 4, 
    0.66667, 1.6729*^8, 1.8965*^8, 
    0.052745}, {"H", "I", "3", "2S -  4p  2Po", Null, 
    972.5368, 0., 2, 6, 1., 6.8189*^7, 1.308*^7, 
    0.029003}, {"H", "I", "3", "2S -  4p  2Po", Null, 972.5371, 0., 
    2, 2, 0.33333, 6.8179*^7, 8.1259*^7, 
    0.0096677}, {"H", "I", "3", "2S -  4p  2Po", Null, 972.5367, 
    0., 2, 4, 0.66667, 6.8189*^7, 
    8.126*^7, 0.019336}, 
   {"H", "I", "4", "2S -  5p  2Po", Null, 949.7431, 0., 2, 6, 1., 
    3.4379*^7, 4.2031*^7, 0.013943}, 
   {"H", "I", "5", "2S -  6p  2Po", Null, 937.8035, 0., 2, 6, 1., 1.973*^7, 
    2.45*^7, 0.0078042}, 
   {"H", "I", "6", "2S -  7p  2Po", Null, 930.7483, 0., 2, 6, 1., 
    1.2369*^7, 0., 0.0048157}, 
   {"H", "I", "7", "2S -  8p  2Po", Null, 926.2257, 0., 2, 6, 1., 
    8.249*^6, 0., 0.0031828}, {"H", "I", "8", "2S -  9p  2Po", Null, 
    923.1504, 0., 2, 6, 1., 5.782*^6, 0., 0.0022162}, 
   {"H", "I", "9", "2S - 10p  2Po", Null, 920.9631, 0., 2, 6, 1., 
    4.208*^6, 0., 0.0016052}, 
   {"H", "I", "10", "2S - 11p  2Po", Null, 919.3514, 0., 2, 6, 1., 
    3.1589*^6, 0., 0.0012005}, 
   {"H", "I", "11", "2S - 12p  2Po", Null, 918.1294, 0., 2, 6, 1., 
    2.43*^6, 0., 0.00092128}, 
   {"H", "I", "12", "2S - 13p  2Po", Null, 917.1806, 0., 2, 6, 1., 1.91*^6, 
    0., 0.00072264}, 
   {"D", "I", "1", "2S -  2p  2Po", Null, 1215.3394, 0., 2, 6, 1., 
    6.27*^8, 0., 0.41652}, 
   {"D", "I", "1", "2S -  2p  2Po", Null, 1215.343, 0., 2, 2, 0.33333, 
    6.2699*^8, 6.2699*^8, 0.13884}, 
   {"D", "I", "1", "2S -  2p  2Po", Null, 1215.3376, 0., 2, 4, 
    0.66667, 6.2701*^8, 6.2701*^8, 0.27768}, 
   {"D", "I", "2", "2S -  3p  2Po", Null, 1025.4433, 0., 2, 6, 1., 
    1.673*^8, 2.245*^7, 0.079122}, 
   {"D", "I", "2", "2S -  3p  2Po", Null, 1025.444, 0., 2, 2, 
    0.33333, 1.673*^8, 1.8975*^8, 0.026374}, 
   {"D", "I", "2", "2S -  3p  2Po", Null, 1025.4429, 0., 2, 4, 
    0.66667, 1.673*^8, 1.8975*^8, 0.052748}, 
   {"D", "I", "3", "2S -  4p  2Po", Null, 972.2722, 0., 2, 6, 1., 
    6.8249*^7, 1.308*^7, 0.029013}, 
   {"D", "I", "3", "2S -  4p  2Po", Null, 972.2725, 0., 2, 2, 0.33333, 
    6.8239*^7, 8.1319*^7, 0.0096709}, 
   {"D", "I", "3", "2S -  4p  2Po", Null, 972.2721, 0., 2, 4, 
    0.66667, 6.8249*^7, 8.132*^7, 
    0.019342}, {"D", "I", "4", "2S -  5p  2Po", Null, 949.4847, 
    0., 2, 6, 1., 3.4499*^7, 4.2061*^7, 0.013948}, 
   {"D", "I", "5", "2S -  6p  2Po", Null, 937.5484, 0., 2, 6, 1., 1.975*^7, 
    2.4529*^7, 0.0078079}, 
   {"D", "1", "6", "2S -  7p  2Po", Null, 930.4951, 0., 2, 6, 1., 
    1.237*^7, 0., 0.004817}, {"D", "I", "7", "2S -  8p  2Po", Null, 
    925.9737, 0., 2, 6, 1., 8.256*^6, 0., 0.0031838}, 
   {"Li", "I", "1v", "2S -  2p  2Po", 6707.812, 6709.6633, 
    0., 2, 6, 1., 3.664*^7, 0., 0.74188}, 
   {"Li", "I", "1v", "2S -  2p  2Po", 6707.9121, 6709.764, 
    0., 2, 2, 0.33333, 3.6638*^7, 3.6638*^7, 0.24729}, 
   {"Li", "I", "1v", "2S -  2p  2Po", 6707.7612, 6709.613, 0., 2, 4, 
    0.66667, 3.6641*^7, 3.6641*^7, 0.49459}, 
   {"Li", "I", "2v", "2S -  3p  2Po", 3232.631, 3233.564, 0., 2, 6, 
    1., 1.1799*^6, 4.9499*^6, 0.0055021}, 
   {"Li", "I", "1", "2S -  4p  2Po", 2741.184, 2741.995, 0., 2, 6, 1., 
    1.33*^6, 2.6*^6, 0.00449748}, 
   {"Li", "I", "2", "2S -  5p  2Po", 2562.302, 2563.07, 0., 2, 6, 1., 
    914000., 1.7159*^6, 0.0027005}, 
   {"Li", "I", "3", "2S -  6p  2Po", 2475.0322, 2475.78, 0., 2, 6, 1., 
    616999.9999999999, 1.427*^6, 0.0017009}, 
   {"Be", "I", "1", "1S -  2p  1Po", 2348.61, 2349.329, 0., 1, 3, 
    1., 5.4699*^8, 5.4699*^8, 1.3554}, 
   {"Be", "I", "", "1S -  3p  1Po", Null, 1661.478, 0., 1, 3, 1., 2.*^7, 0., 
    0.024831}, {"Be", "II", "1v", "2S -  2p  2Po", 3130.635, 
    3131.5423, 0., 2, 6, 1., 1.1599*^8, 0., 0.50722}, 
   {"Be", "II", "1v", "2S -  2p  2Po", 3131.0654, 3131.973, 
    0., 2, 2, 0.33333, 1.1495*^8, 1.1495*^8, 0.16905}, 
   {"Be", "II", "1v", "2S -  2p  2Po", 3130.42, 3131.327, 
    0., 2, 4, 0.66667, 1.1502*^8, 
    1.1502*^8, 0.33817}, 
   {"Be", "II", "1", "2S -  3p  2Po", Null, 1036.3057, 0., 2, 6, 1., 
    1.66*^8, 1.28*^7, 0.080179}, 
   {"Be", "II", "1", "2S -  3p  2Po", Null, 1036.319, 0., 2, 2, 
    0.33333, 1.6599*^8, 1.7879*^8, 0.026726}, 
   {"Be", "II", "1", "2S -  3p  2Po", Null, 1036.299, 0., 2, 4, 
    0.66667, 1.66*^8, 1.788*^8, 0.053453}, 
   {"B", "I", "1", "2Po-  3s  2S ", 2497.406, 2498.1588, 10.169, 6, 2, 1., 
    2.54*^8, 0., 0.079215}, 
   {"B", "I", "1", "2Po-  3s  2S ", 2497.7231, 2498.4762, 
    15.254, 4, 2, 0.66667, 1.6927*^8, 
    2.54*^8, 0.079206}, {"B", "I", "1", "2Po-  3s  2S ", 2496.7714, 
    2497.5243, 0., 2, 2, 0.33333, 8.473*^7, 2.54*^8, 
    0.079235}, {"B", "I", "2", "2Po-  2p2 2D ", 
    2089.333, 2089.9975, 10.169, 6, 10, 1., 4.24*^7, 0., 
    0.046277}, {"B", "I", "2", "2Po-  2p2 2D ", 2089.5634, 
    2090.228, 15.254, 4, 4, 0.066679, 7.0647*^6, 
    4.2429*^7, 0.0046274}, 
   {"B", "I", "2", "2Po-  2p2 2D ", 2089.5493, 2090.214, 15.254, 
    4, 6, 0.6, 4.2387*^7, 4.2387*^7, 0.041645}, 
   {"B", "I", "2", "2Po-  2p2 2D ", 2088.8974, 2089.562, 
    0., 2, 4, 0.33333, 3.5355*^7, 4.2429*^7, 0.046286}, 
   {"B", "I", "3", "2Po-  3d  2D ", Null, 1826.2317, 10.169, 6, 10, 1., 
    2.4999*^8, 1.3899*^7, 0.2}, 
   {"B", "I", "3", "2Po-  3d  2D ", Null, 1826.4047, 
    15.254, 4, 4, 0.066679, 3.9991*^7, 
    2.539*^8, 0.019999}, {"B", "I", "3", "2Po-  3d  2D ", Null, 
    1826.399, 15.254, 4, 6, 0.6, 2.3993*^8, 2.5373*^8, 
    0.17998}, {"B", "I", "3", "2Po-  3d  2D ", Null, 1825.896, 0., 
    2, 4, 0.33333, 2.0011*^8, 2.539*^8, 0.20003}, 
   {"B", "I", "Auto", "2Po-  2p2 2P ", Null, 1378.921, 10.169, 6, 6, 1., 
    2.1*^9, 0., 0.59862}, {"B", "I", "Auto", "2Po-  2p2 2P ", Null, 
    1379.165, 15.254, 4, 2, 0.11111, 
    6.9962*^8, 0., 0.099752}, 
   {"B", "I", "Auto", "2Po-  2p2 2P ", Null, 1378.944, 
    15.254, 4, 4, 0.55556, 1.7499*^9, 0., 0.49885}, 
   {"B", "I", "Auto", "2Po-  2p2 2P ", Null, 1378.875, 0., 2, 2, 
    0.22222, 1.4001*^9, 0., 0.39909}, 
   {"B", "I", "Auto", "2Po-  2p2 2P ", Null, 1378.654, 0., 2, 4, 
    0.11111, 3.5029*^8, 0., 0.19958}, 
   {"B", "II", "1", "1S -  2p  1Po", Null, 1362.461, 0., 1, 3, 1., 
    1.1829*^9, 1.1829*^9, 0.98683}, 
   {"B", "III", "", "2S -  2p  2Po", 2066.2612, 2066.9214, 0., 2, 6, 1., 
    1.91*^8, 0., 0.36699}, {"B", "III", "", "2S -  2p  2Po", 2067.233, 
    2067.893, 0., 2, 2, 0.33333, 1.9073*^8, 1.9073*^8, 0.12227}, 
   {"B", "III", "", "2S -  2p  2Po", 2065.776, 2066.436, 0., 2, 4, 
    0.66667, 1.9114*^8, 1.9114*^8, 0.24472}, 
   {"C", "I", "1", "3P -  2p3 5So", 2967.2143, 2968.081, 
    43.499, 5, 5, Null, 16.479, 0., 
    2.1752*^-8}, {"C", "I", "1", "3P -  2p3 5So", 2964.8391, 
    2965.705, 16.499, 3, 5, Null, 7.721, 0., 1.6968*^-8}, 
   {"C", "I", "2", "3P -  3s  3Po", Null, 1657.1816, 
    29.577, 9, 9, 1., 3.41*^8, 0., 0.14039}, 
   {"C", "I", "2", "3P -  3s  3Po", Null, 1658.1212, 43.499, 5, 
    3, 0.13889, 1.4184*^8, 3.4078*^8, 0.035079}, 
   {"C", "I", "2", "3P -  3s  3Po", Null, 1657.9068, 
    16.499, 3, 1, 0.11111, 3.4055*^8, 
    3.4055*^8, 0.046777}, 
   {"C", "I", "2", "3P -  3s  3Po", Null, 1657.3792, 16.499, 3, 
    3, 0.08333, 8.5216*^7, 3.4078*^8, 0.035093}, 
   {"C", "I", "2", "3P -  3s  3Po", Null, 1657.0082, 
    43.499, 5, 5, 0.41667, 2.5583*^8, 
    3.4122*^8, 0.10531}, 
   {"C", "I", "2", "3P -  3s  3Po", Null, 1656.9283, 0., 1, 3, 
    0.11111, 1.1372*^8, 3.4078*^8, 0.14041}, 
   {"C", "I", "2", "3P -  3s  3Po", Null, 1656.2672, 
    16.499, 3, 5, 0.13889, 8.53927*^7, 
    3.4122*^8, 0.058531}, 
   {"C", "I", "2.01", "3P -  3s  1Po", Null, 1614.5068, 43.499, 
    5, 3, Null, 40410., 3.6352*^8, 
    9.4749*^-6}, {"C", "I", "2.01", "3P -  3s  1Po", Null, 
    1613.8033, 16.499, 3, 3, Null, 34130., 3.6352*^8, 
    0.000013326}, {"C", "I", "2.01", "3P -  3s  1Po", Null, 
    1613.3763, 0., 1, 3, Null, 47390.9, 
    3.6352*^8, 0.000055488}, 
   {"C", "I", "3", "3P -  2p3 3Do", Null, 1561.0544, 
    29.577, 9, 15, 1., 1.32*^8, 0., 
    0.080374}, {"C", "I", "3", "3P -  2p3 3Do", Null, 1561.4384, 
    43.499, 5, 7, 0.46667, 1.319*^8, 1.319*^8, 
    0.067498}, {"C", "I", "3", "3P -  2p3 3Do", Null, 1561.3667, 
    43.499, 5, 3, 0.0055699, 3.6674*^6, 
    1.3214*^8, 0.00080422}, 
   {"C", "I", "3", "3P -  2p3 3Do", Null, 1561.3402, 43.499, 5, 
    5, 0.08333, 3.2981*^7, 1.3205*^8, 0.012053}, 
   {"C", "I", "3", "3P -  2p3 3Do", Null, 1560.709, 
    16.499, 3, 3, 0.08333, 5.5034*^7, 1.3214*^8, 
    0.020097}, {"C", "I", "3", "3P -  2p3 3Do", Null, 1560.6822, 
    16.499, 3, 5, 0.25, 9.9071*^7, 1.3205*^8, 
    0.060295}, {"C", "I", "3", "3P -  2p3 3Do", Null, 
    1560.3092, 0., 1, 3, 0.11111, 7.3438*^7, 
    1.3214*^8, 0.080412}, 
   {"C", "I", "4", "3P -  2p3 3Po", Null, 1329.3392, 29.577, 9, 
    9, 1., 2.1999*^8, 1.21*^6, 0.058019}, 
   {"C", "I", "4", "3P -  2p3 3Po", Null, 1329.6005, 
    43.499, 5, 3, 0.13889, 9.1197*^7, 
    2.2026*^8, 0.014502}, 
   {"C", "I", "4", "3P -  2p3 3Po", Null, 1329.5775, 43.499, 5, 
    5, 0.41667, 1.6416*^8, 2.2015*^8, 
    0.043507}, {"C", "I", "4", "3P -  2p3 3Po", Null, 1329.1233, 
    16.499, 3, 3, 0.08333, 5.4774*^7, 2.2026*^8, 
    0.014507}, {"C", "I", "4", "3P -  2p3 3Po", Null, 1329.1004, 
    16.499, 3, 5, 0.13889, 5.4789*^7, 
    2.2015*^8, 0.024179}, 
   {"C", "I", "4", "3P -  2p3 3Po", Null, 1329.0853, 
    16.499, 3, 1, 0.11111, 2.1912*^8, 
    2.2033*^8, 0.019343}, 
   {"C", "I", "4", "3P -  2p3 3Po", Null, 1328.8333, 0., 1, 3, 
    0.11111, 7.3083*^7, 2.2026*^8, 0.058041}, 
   {"C", "I", "4.01", "3P -  3d  1Do", Null, 1288.0553, 43.499, 
    5, 5, Null, 56750.9, 5.9911*^7, 0.000014115}, 
   {"C", "I", "4.01", "3P -  3d  1Do", Null, 1287.6075, 43.499, 
    3, 5, Null, 154000., 5.9911*^7, 0.0000637968}, 
   {"C", "I", "5", "3P -  4s  3Po", Null, 1280.3561, 
    29.577, 9, 9, 1., 8.9298*^7, 2.1399*^7, 
    0.021946}, {"C", "I", "5", "3P -  4s  3Po", Null, 1280.847, 
    43.499, 5, 3, 0.12781, 3.42*^7, 1.0634*^8, 
    0.0050478}, {"C", "I", "5", "3P -  4s  3Po", Null, 
    1280.597, 16.499, 3, 1, 0.10387, 
    8.3429*^7, 1.0473*^8, 0.0068372}, 
   {"C", "I", "5", "3P -  4s  3Po", Null, 1280.4042, 
    16.499, 3, 3, 0.066598, 1.7837*^7, 
    1.0634*^8, 0.0043841}, 
   {"C", "I", "5", "3P -  4s  3Po", Null, 1280.3328, 
    43.499, 5, 5, 0.37101, 5.9639*^7, 1.1434*^8, 
    0.014656}, {"C", "I", "5", "3P -  4s  3Po", Null, 1280.1353, 0., 
    1, 3, 0.12312, 3.3999*^7, 1.0634*^8, 
    0.024322}, {"C", "I", "5", "3P -  4s  3Po", Null, 
    1279.8904, 16.499, 3, 5, 0.2076, 
    3.3406*^7, 1.1434*^8, 0.013673}, 
   {"C", "I", "6", "3P -  3d  3Fo", Null, 1279.4977, 
    43.499, 5, 5, Null, 80000., 2.878*^7, 0.000019635}, 
   {"C", "I", "6", "3P -  3d  3Fo", Null, 1279.2286, 
    43.499, 5, 7, Null, 1.1*^7, 3.79*^7, 0.0037781}, 
   {"C", "I", "6", "3P -  3d  3Fo", Null, 1279.0558, 16.499, 3, 
    5, Null, 1.8*^6, 2.878*^7, 0.0007358}, 
   {"C", "I", "7", "3P -  3d  3Do", Null, 1277.4626, 29.577, 9, 
    15, 1., 2.37*^8, 2.1999*^7, 0.096638}, 
   {"C", "I", "7", "3P -  3d  3Do", Null, 1277.9538, 
    43.499, 5, 3, 0.0055699, 6.581*^6, 2.5895*^8, 
    0.00096679}, {"C", "I", "7", "3P -  3d  3Do", Null, 
    1277.7229, 43.499, 5, 5, 0.08333, 5.9211*^7, 
    2.5894*^8, 0.014492}, 
   {"C", "I", "7", "3P -  3d  3Do", Null, 1277.5496, 
    43.499, 5, 7, 0.46667, 2.3695*^8, 
    2.5885*^8, 0.081171}, 
   {"C", "I", "7", "3P -  3d  3Do", Null, 1277.513, 
    16.499, 3, 3, 0.08333, 9.8734*^7, 2.5895*^8, 
    0.024158}, {"C", "I", "7", "3P -  3d  3Do", Null, 
    1277.2823, 16.499, 3, 5, 0.25, 1.7783*^8, 
    2.5894*^8, 0.072489}, 
   {"C", "I", "7", "3P -  3d  3Do", Null, 1277.2454, 0., 1, 3, 
    0.11111, 1.3173*^8, 2.5895*^8, 0.096654}\
, {"C", "I", "7.01", 
      "3P -  4s  1Po", Null, 1277.1901, 
    43.499, 5, 3, Null, 2.321*^6, 8.7904*^7, 
    0.00034056}, {"C", "I", "7.01", "3P -  4s  1Po", Null, 
    1276.7498, 16.499, 3, 3, Null, 1.1749*^7, 
    8.7904*^7, 0.0028699}, 
   {"C", "I", "7.01", "3P -  4s  1Po", Null, 1276.4825, 0., 1, 3, 
    Null, 6.1439*^6, 8.7904*^7, 0.0045018}, 
   {"C", "I", "8", "3P -  3d  1Fo", Null, 1274.109, 
    43.499, 5, 7, Null, 1.439*^6, 1.9384*^8, 
    0.00049038}, {"C", "I", "8.01", "3P -  3d  1Po", Null, 
    1270.8439, 43.499, 5, 3, Null, 7317., 1.7171*^8, 
    1.063*^-6}, {"C", "I", "8.01", "3P -  3d  1Po", Null, 1270.408, 
    16.499, 3, 3, Null, 270400., 1.7171*^8, 
    0.000065426}, {"C", "I", "8.01", "3P -  3d  1Po", Null, 
    1270.1434, 0., 1, 3, Null, 535099.9999999999, 
    1.7171*^8, 0.00038826}, 
   {"C", "I", "9", "3P -  3d  3Po", Null, 1261.2681, 29.577, 9, 
    9, 1., 1.6599*^8, 2.1799*^7, 0.039351}, 
   {"C", "I", "9", "3P -  3d  3Po", Null, 1261.5519, 
    43.499, 5, 5, 0.41667, 1.2367*^8, 
    1.8663*^8, 0.029507}, 
   {"C", "I", "9", "3P -  3d  3Po", Null, 1261.4257, 
    43.499, 5, 3, 0.13889, 6.8725*^7, 
    1.8677*^8, 0.0098366}, 
   {"C", "I", "9", "3P -  3d  3Po", Null, 1261.1223, 
    16.499, 3, 5, 0.13889, 4.1265*^7, 1.8663*^8, 
    0.016398}, {"C", "I", "9", "3P -  3d  3Po", Null, 
    1260.9962, 16.499, 3, 3, 0.08333, 4.1275*^7, 
    1.8677*^8, 0.0098395}, 
   {"C", "I", "9", "3P -  3d  3Po", Null, 1260.9266, 
    16.499, 3, 1, 0.11111, 1.6513*^8, 
    1.8683*^8, 0.01312}, 
   {"C", "I", "9", "3P -  3d  3Po", Null, 1260.7355, 0., 1, 3, 
    0.11111, 5.5069*^7, 1.8677*^8, 
    0.039367}, {"C", "I", "9.01", "3P -  4d  1Do", Null, 
    1198.2618, 43.499, 5, 5, Null, 45506., 0., 
    9.7956*^-6}, {"C", "I", "9.01", "3P -  4d  1Do", Null, 1197.8742, 
    16.499, 3, 5, Null, 10.675, 0., 
    3.8272*^-9}, {"C", "I", "9.02", "3P -  5s  3Po", Null, 
    1194.1308, 29.577, 9, 9, 1., 4.4*^7, 0., 0.0094062}, 
   {"C", "I", "9.02", "3P -  5s  3Po", Null, 1194.6145, 
    43.499, 5, 3, 0.13889, 1.8311*^7, 0., 0.0023506}, 
   {"C", "I", "9.02", "3P -  5s  3Po", Null, 1194.4055, 16.499, 
    3, 1, 0.11111, 4.3969*^7, 0., 0.0031346}\
, {"C", "I", "9.02", 
      "3P -  5s  3Po", Null, 1194.2293, 
    16.499, 3, 3, 0.08333, 1.0997*^7, 0., 0.0023513}, 
   {"C", "I", "9.02", "3P -  5s  3Po", Null, 1194.0635, 
    43.499, 5, 5, 0.41667, 3.3006*^7, 0., 
    0.0070551}, {"C", "I", "9.02", "3P -  5s  3Po", Null, 
    1193.9955, 0., 1, 3, 0.11111, 1.4672*^7, 0., 
    0.0094071}, {"C", "I", "9.02", "3P -  5s  3Po", Null, 
    1193.6787, 16.499, 3, 5, 0.13889, 
    1.1013*^7, 0., 0.0039208}, 
   {"C", "I", "10", "3P -  4d  3Fo", Null, 1194.6862, 43.499, 5, 
    5, Null, 1700.8, 0., 3.6394*^-7}, 
   {"C", "I", "10", "3P -  4d  3Fo", Null, 1194.4882, 
    43.499, 5, 7, Null, 30527., 0., 9.1418*^-6}, 
   {"C", "I", "10", "3P -  4d  3Fo", Null, 1194.3009, 
    16.499, 3, 5, Null, 10494., 0., 3.7401*^-6}, 
   {"C", "I", "11", "3P -  4d  3Do", Null, 1193.1761, 
    29.577, 9, 15, 1., 1.25*^8, 5.15*^6, 0.044466}, 
   {"C", "I", "11", "3P -  4d  3Do", Null, 1193.6489, 
    43.499, 5, 3, 0.0055699, 3.4709*^6, 1.3016*^8, 
    0.00044484}, {"C", "I", "11", "3P -  4d  3Do", Null, 
    1193.3932, 43.499, 5, 5, 0.08333, 3.1232*^7, 1.3017*^8, 
    0.00666848}, {"C", "I", "11", "3P -  4d  3Do", Null, 
    1193.2643, 16.499, 3, 3, 0.08333, 5.207*^7, 1.3016*^8, 
    0.011115}, {"C", "I", "11", "3P -  4d  3Do", Null, 1193.2401, 
    43.499, 5, 7, 0.46667, 1.2498*^8, 
    1.3013*^8, 0.037349}, 
   {"C", "I", "11", "3P -  4d  3Do", Null, 1193.0308, 0., 1, 3, 
    0.11111, 6.9469*^7, 1.3016*^8, 0.0444718}\
, {"C", "I", "11", 
      "3P -  4d  3Do", Null, 1193.0088, 16.499, 3, 
    5, 0.25, 9.3789*^7, 1.3017*^8, 0.033354}, 
   {"C", "I", "12", "3P -  5s  1Po", Null, 1192.8347, 43.499, 5, 
    3, Null, Null, Null, Null}, 
   {"C", "I", "12", "3P -  5s  1Po", Null, 1192.4507, 16.499, 3, 
    3, Null, 9.8669*^6, 0., 0.0021032}, 
   {"C", "I", "12", "3P -  5s  1Po", Null, 1192.2175, 0., 1, 3, 
    Null, 4.1144*^6, 0., 0.0026303}, 
   {"C", "I", "13", "3P -  4d  1Fo", Null, 1191.838, 
    43.499, 5, 7, Null, 1890.6, 0., 
    5.6368*^-7}, {"C", "I", "13.01", "3P -  4d  1Po", Null, 
    1190.6357, 43.499, 5, 3, Null, 5.6946*^6, 0., 
    0.00072616}, {"C", "I", "13.01", "3P -  4d  1Po", Null, 
    1190.253, 16.499, 3, 3, Null, 2.8559*^6, 0., 
    0.00060657}, {"C", "I", "13.01", "3P -  4d  1Po", Null, 
    1190.0207, 0., 1, 3, Null, 5.9692*^6, 0., 
    0.0038019}, {"C", "I", "14", "3P -  4d  3Po", Null, 
    1189.3454, 29.577, 9, 9, 1., 7.9*^7, 0., 0.016753}, 
   {"C", "I", "14", "3P -  4d  3Po", Null, 1189.6307, 
    43.499, 5, 5, 0.41667, 5.9208*^7, 0., 0.012562}, 
   {"C", "I", "14", "3P -  4d  3Po", Null, 1189.4469, 
    43.499, 5, 3, 0.13889, 3.2909*^7, 0., 
    0.0041889}, {"C", "I", "14", "3P -  4d  3Po", Null, 
    1189.2487, 16.499, 3, 5, 0.13889, 1.9755*^7, 0., 
    0.0069812}, {"C", "I", "14", "3P -  4d  3Po", Null, 1189.065, 
    16.499, 3, 3, 0.08333, 1.9763*^7, 0., 
    0.0041891}, {"C", "I", "14", "3P -  4d  3Po", Null, 
    1188.9925, 16.499, 3, 1, 0.11111, 7.907*^7, 0., 
    0.005586}, {"C", "I", "14", "3P -  4d  3Po", Null, 
    1188.8332, 0., 1, 3, 0.11111, 2.6367*^7, 0., 
    0.016769}, {"C", "I", "14.01", "3P -  5d  1Do", Null, 
    1160.8766, 43.499, 5, 5, Null, 403280., 0., 
    0.000081476}, {"C", "I", "14.01", "3P -  5d  1Do", Null, 1160.5129, 
    16.499, 3, 5, Null, 176150., 0., 0.000059276}, 
   {"C", "I", "15", "3P -  5d  3Fo", Null, 1159.0945, 
    43.499, 5, 5, Null, 198120., 0., 0.000039905}, 
   {"C", "I", "15", "3P -  5d  3Fo", Null, 1158.9666, 
    43.499, 5, 7, Null, 4.4761*^6, 0., 0.0012619}, 
   {"C", "I", "15", "3P -  5d  3Fo", Null, 1158.7319, 
    16.499, 3, 5, Null, 5.5874*^6, 0., 0.0018745}, 
   {"C", "I", "15.01", "3P -  6s  3Po", Null, 1158.4523, 
    29.577, 9, 9, 1., 1.7*^7, 0., 0.0034203}, 
   {"C", "I", "15.01", "3P -  6s  3Po", Null, 1158.9066, 
    43.499, 5, 3, 0.13889, 7.0751*^6, 0., 0.00085474}, 
   {"C", "I", "15.01", "3P -  6s  3Po", Null, 1158.6742, 
    16.499, 3, 1, 0.11111, 1.699*^7, 0., 
    0.0011399}, {"C", "I", "15.01", "3P -  6s  3Po", Null, 
    1158.5441, 16.499, 3, 3, 0.08333, 4.2488*^6, 0., 
    0.00085497}, {"C", "I", "15.01", "3P -  6s  3Po", Null, 
    1158.3969, 43.499, 5, 5, 0.41667, 
    1.2752*^7, 0., 0.0025654}, 
   {"C", "I", "15.01", "3P -  6s  3Po", Null, 1158.324, 0., 1, 3, 
    0.11111, 5.6685*^6, 0., 0.0034206}, 
   {"C", "I", "15.01", "3P -  6s  3Po", Null, 1158.0347, 
    16.499, 3, 5, 0.13889, 4.2546*^6, 0., 0.0014256}, 
   {"C", "I", "16", "3P -  5d  3Do", Null, 1157.9655, 
    29.577, 9, 15, 1., 6.5*^7, 0., 0.021778}, 
   {"C", "I", "16", "3P -  5d  3Do", Null, 1158.4919, 43.499, 5, 
    3, 0.0055699, 1.8045*^6, 0., 0.00021785}, 
   {"C", "I", "16", "3P -  5d  3Do", Null, 1158.1315, 43.499, 5, 
    5, 0.08333, 1.6242*^7, 0., 0.0032669}, 
   {"C", "I", "16", "3P -  5d  3Do", Null, 1158.1296, 16.499, 3, 
    3, 0.08333, 2.7071*^7, 0., 0.0054434}, 
   {"C", "I", "16", "3P -  5d  3Do", Null, 1158.0186, 43.499, 5, 
    7, 0.46667, 6.4992*^7, 0., 0.018292}, 
   {"C", "I", "16", "3P -  5d  3Do", Null, 1157.9097, 0., 1, 3, 
    0.11111, 3.6116*^7, 0., 0.021778}, 
   {"C", "I", "16", "3P -  5d  3Do", Null, 1157.7695, 16.499, 3, 
    5, 0.25, 4.8775*^7, 0., 0.016336}, 
   {"C", "I", "17", "3P -  6s  1Po", Null, 1157.7672, 
    43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "17", "3P -  6s  1Po", Null, 1157.4054, 16.499, 3, 
    3, Null, 4.4674*^6, 0., 0.00089718}, 
   {"C", "I", "17", "3P -  6s  1Po", Null, 1157.1857, 0., 1, 3, 
    Null, 912460., 0., 0.00054954}, 
   {"C", "I", "18", "3P -  5d  1Fo", Null, 1157.3299, 43.499, 5, 
    7, Null, 2.3021*^6, 0., 0.00064719}, 
   {"C", "I", "18.01", "3P -  5d  1Po", Null, 1156.7646, 43.499, 
    5, 3, Null, 4.271*^6, 0., 0.00051408}, 
   {"C", "I", "18.01", "3P -  5d  1Po", Null, 1156.4035, 
    16.499, 3, 3, Null, 7.7768*^6, 0., 
    0.0015591}, {"C", "I", "18.01", "3P -  5d  1Po", Null, 
    1156.1842, 0., 1, 3, Null, 46878., 0., 0.000028184}, 
   {"C", "I", "19", "3P -  5d  3Po", Null, 1156.2939, 
    29.577, 9, 9, 1., 8.6999*^7, 0., 
    0.017238}, {"C", "I", "19", "3P -  5d  3Po", Null, 
    1156.5601, 43.499, 5, 5, 0.41667, 
    6.44568*^7, 0., 0.012926}, 
   {"C", "I", "19", "3P -  5d  3Po", Null, 1156.3893, 43.499, 5, 
    3, 0.13889, 3.5825*^7, 0., 0.0043092}, 
   {"C", "I", "19", "3P -  5d  3Po", Null, 1156.199, 
    16.499, 3, 5, 0.13889, 2.1505*^7, 0., 0.0071832}, 
   {"C", "I", "19", "3P -  5d  3Po", Null, 1156.0283, 16.499, 3, 
    3, 0.08333, 2.1514*^7, 0., 0.0043104}, 
   {"C", "I", "19", "3P -  5d  3Po", Null, 1155.979, 16.499, 3, 
    1, 0.11111, 8.6069*^7, 0., 0.0057476}, 
   {"C", "I", "19", "3P -  5d  3Po", Null, 1155.8092, 0., 1, 3, 
    0.11111, 2.8702*^7, 0., 0.017245}, 
   {"C", "I", "20", "3P -  6d  1Do", Null, 1141.6783, 43.499, 5, 
    5, Null, 218820., 0., 0.000042759}, 
   {"C", "I", "20", "3P -  6d  1Do", Null, 1141.3265, 
    16.499, 3, 5, Null, 79498., 0., 0.000025875}, 
   {"C", "I", "21", "3P -  6d  3Fo", Null, 1140.7086, 
    43.499, 5, 5, Null, 126130., 0., 0.000024605}, 
   {"C", "I", "21", "3P -  6d  3Fo", Null, 1140.6413, 
    43.499, 5, 7, Null, 3.1243*^6, 0., 0.00085316}, 
   {"C", "I", "21", "3P -  6d  3Fo", Null, 1140.3573, 
    16.499, 3, 5, Null, 6.6236*^6, 0., 0.0021522}, 
   {"C", "I", "21.01", "3P -  7s  3Po", Null, 1140.1505, 29.577, 
    9, 9, 1., 9.*^6, 0., 0.001754}, 
   {"C", "I", "21.01", "3P -  7s  3Po", Null, 1140.5739, 43.499, 
    5, 3, 0.13889, 3.7459*^6, 0., 0.00043833}, 
   {"C", "I", "21.01", "3P -  7s  3Po", Null, 1140.3163, 
    16.499, 3, 1, 0.11111, 8.9968*^6, 0., 
    0.00058457}, {"C", "I", "21.01", "3P -  7s  3Po", Null, 
    1140.2228, 16.499, 3, 3, 0.08333, 2.2495*^6, 0., 
    0.00043845}, {"C", "I", "21.01", "3P -  7s  3Po", Null, 1140.1166, 
    43.499, 5, 5, 0.41667, 6.7507*^6, 0., 
    0.0013155}, {"C", "I", "21.01", "3P -  7s  3Po", Null, 
    1140.0096, 0., 1, 3, 0.11111, 3.0011*^6, 0., 0.0017542}, 
   {"C", "I", "21.01", "3P -  7s  3Po", Null, 1139.7657, 
    16.499, 3, 5, 0.13889, 2.2523*^6, 0., 0.00073108}, 
   {"C", "I", "22", "3P -  6d  3Do", Null, 1139.759, 
    29.577, 9, 15, 1., 4.3999*^7, 0., 
    0.013957}, {"C", "I", "22", "3P -  6d  3Do", Null, 
    1140.357, 43.499, 5, 3, 0.0055699, 
    1.1935*^6, 0., 0.00013961}, 
   {"C", "I", "22", "3P -  6d  3Do", Null, 1140.006, 16.499, 3, 
    3, 0.08333, 1.7904*^7, 0., 0.0034884}, 
   {"C", "I", "22", "3P -  6d  3Do", Null, 1139.865, 
    43.499, 5, 5, 0.08333, 1.0747*^7, 0., 0.0020933}, 
   {"C", "I", "22", "3P -  6d  3Do", Null, 1139.812, 
    43.499, 5, 7, 0.46667, 4.2994*^7, 0., 0.011724}, 
   {"C", "I", "22", "3P -  6d  3Do", Null, 1139.793, 0., 1, 3, 
    0.11111, 2.3887*^7, 0., 0.013957}, 
   {"C", "I", "22", "3P -  6d  3Do", Null, 1139.5143, 16.499, 3, 
    5, 0.25, 3.2271*^7, 0., 0.010479}, 
   {"C", "I", "22.01", "3P -  7s  1Po", Null, 1139.6501, 43.499, 
    5, 3, Null, Null, Null, Null}, 
   {"C", "I", "22.01", "3P -  7s  1Po", Null, 1139.2995, 16.499, 
    3, 3, Null, Null, Null, Null}, 
   {"C", "I", "22.01", "3P -  7s  1Po", Null, 1139.0867, 0., 1, 3, Null, 
    1.1855*^6, 0., 0.00069183}, 
   {"C", "I", "22.02", "3P -  6d  1Fo", Null, 1139.4255, 43.499, 
    5, 7, Null, 1.5335*^6, 0., 0.00041786}, 
   {"C", "I", "22.03", "3P -  6d  1Po", Null, 1139.1238, 43.499, 
    5, 3, Null, 383599.9999999999, 0., 0.000044774}, 
   {"C", "I", "22.03", "3P -  6d  1Po", Null, 1138.7736, 
    16.499, 3, 3, Null, 765.86, 0., 
    1.4889*^-7}, {"C", "I", "22.03", "3P -  6d  1Po", Null, 
    1138.5609, 0., 1, 3, Null, 2.0621*^6, 0., 0.0012023}, 
   {"C", "I", "23", "3P -  6d  3Po", Null, 1138.8444, 
    29.577, 9, 9, 1., 1.*^7, 0., 0.0019444}, 
   {"C", "I", "23", "3P -  6d  3Po", Null, 1139.0931, 
    43.499, 5, 5, 0.41667, 7.49518*^6, 0., 
    0.0014589}, {"C", "I", "23", "3P -  6d  3Po", Null, 
    1138.947, 43.499, 5, 3, 0.13889, 4.1656*^6, 0., 
    0.00048606}, {"C", "I", "23", "3P -  6d  3Po", Null, 
    1138.7428, 16.499, 3, 5, 0.13889, 2.5007*^6, 0., 
    0.00081025}, {"C", "I", "23", "3P -  6d  3Po", Null, 
    1138.597, 16.499, 3, 3, 0.08333, 2.5015*^6, 0., 
    0.00048619}, {"C", "I", "23", "3P -  6d  3Po", Null, 
    1138.557, 16.499, 3, 1, 0.11111, 
    1.0007*^7, 0., 0.00064829}, 
   {"C", "I", "23", "3P -  6d  3Po", Null, 1138.384, 0., 1, 3, 
    0.11111, 3.3373*^6, 0., 0.0019452}, 
   {"C", "I", "23.01", "3P -  7d  1Do", Null, 1130.5155, 43.499, 
    5, 5, Null, 125490., 0., 0.000024045}, 
   {"C", "I", "23.01", "3P -  7d  1Do", Null, 1130.1706, 16.499, 
    3, 5, Null, 41580.9, 0., 0.00001327}, 
   {"C", "I", "24", "3P -  7d  3Fo", Null, 1129.969, 43.499, 5, 
    5, Null, 19455., 0., 3.7242*^-6}, 
   {"C", "I", "24", "3P -  7d  3Fo", Null, 1129.927, 
    43.499, 5, 7, Null, 1.7496*^6, 0., 
    0.00046885}, {"C", "I", "24", "3P -  7d  3Fo", Null, 
    1129.624, 16.499, 3, 5, Null, 3.7958*^6, 0., 
    0.0012103}, {"C", "I", "24.01", "3P -  8s  3Po", Null, 
    1129.4528, 29.577, 9, 9, 1., 3.77*^6, 0., 
    0.000721}, {"C", "I", "24.01", "3P -  8s  3Po", Null, 
    1129.8712, 43.499, 5, 3, 0.13889, 1.5691*^6, 0., 
    0.00018018}, {"C", "I", "24.01", "3P -  8s  3Po", Null, 
    1129.5943, 16.499, 3, 1, 0.11111, 
    3.7685*^6, 0., 0.0002403}, {"C", "I", "24.01", "3P -  8s  3Po", Null, 
    1129.5267, 16.499, 3, 3, 0.08333, 942280., 0., 
    0.00018023}, {"C", "I", "24.01", "3P -  8s  3Po", Null, 
    1129.4221, 43.499, 5, 5, 0.41667, 
    2.8278*^6, 0., 0.00054077}, 
   {"C", "I", "24.01", "3P -  8s  3Po", Null, 1129.3175, 0., 1, 3, 
    0.11111, 1.2571*^6, 0., 0.00072108}, 
   {"C", "I", "24.01", "3P -  8s  3Po", Null, 1129.0777, 16.499, 
    3, 5, 0.13889, 943450., 0., 0.00030052}, 
   {"C", "I", "25", "3P -  7d  3Do", Null, 1129.0925, 29.577, 9, 
    15, 1., 6.3466*^6, 0., 0.0020216}, 
   {"C", "I", "25", "3P -  7d  3Do", Null, 1129.747, 43.499, 5, 
    3, 0.0017899, 56372., 0., 6.4719*^-6}, 
   {"C", "I", "25", "3P -  7d  3Do", Null, 1129.402, 16.499, 3, 
    3, 0.12594, 3.9932*^6, 0., 0.00076362}, 
   {"C", "I", "25", "3P -  7d  3Do", Null, 1129.193, 0., 1, 3, 
    0.26921, 8.5405*^6, 0., 0.0048978}, 
   {"C", "I", "25", "3P -  7d  3Do", Null, 1129.1607, 
    43.499, 5, 5, 0.01514, 288180., 0., 0.000055085}, 
   {"C", "I", "25", "3P -  7d  3Do", Null, 1129.141, 43.499, 5, 
    7, 0.43659, 5.9368*^6, 0., 0.0015887}, 
   {"C", "I", "25", "3P -  7d  3Do", Null, 1128.8166, 
    16.499, 3, 5, 0.15134, 2.8835*^6, 0., 0.00091808}, 
   {"C", "I", "25.01", "3P -  8s  1Po", Null, 1129.0299, 43.499, 
    5, 3, Null, Null, Null, Null}, 
   {"C", "I", "25.01", "3P -  8s  1Po", Null, 1128.6859, 16.499, 
    3, 3, Null, 1.4855*^7, 0., 0.0028371}, 
   {"C", "I", "25.01", "3P -  8s  1Po", Null, 1128.477, 0., 1, 3, 
    Null, 1.5561*^6, 0., 0.00089125}, 
   {"C", "I", "25.02", "3P -  7d  1Fo", Null, 1128.9026, 
    43.499, 5, 7, Null, 898940., 0., 0.00024045}, 
   {"C", "I", "26", "3P -  7d  3Po", Null, 1128.5182, 29.577, 9, 
    9, 1., 1.4799*^7, 0., 0.0028067}, 
   {"C", "I", "26", "3P -  7d  3Po", Null, 1128.752, 
    43.499, 5, 5, 0.41667, 1.1018*^7, 0., 
    0.0021046}, {"C", "I", "26", "3P -  7d  3Po", Null, 
    1128.631, 43.499, 5, 3, 0.13889, 
    6.1232*^6, 0., 0.0007016}, 
   {"C", "I", "26", "3P -  7d  3Po", Null, 1128.408, 
    16.499, 3, 5, 0.13889, 3.6761*^6, 0., 0.0011696}, 
   {"C", "I", "26", "3P -  7d  3Po", Null, 1128.287, 
    16.499, 3, 3, 0.08333, 3.6771*^6, 0., 0.00070178}, 
   {"C", "I", "26", "3P -  7d  3Po", Null, 1128.252, 16.499, 3, 
    1, 0.11111, 1.471*^7, 0., 0.00093577}, 
   {"C", "I", "26", "3P -  7d  3Po", Null, 1128.079, 0., 1, 3, 
    0.11111, 4.9057*^6, 0., 0.0028077}, 
   {"C", "I", "26.01", "3P -  7d  1Po", Null, 1128.724, 43.499, 
    5, 3, Null, Null, Null, Null}, 
   {"C", "I", "26.01", "3P -  7d  1Po", Null, 1128.3801, 
    16.499, 3, 3, Null, Null, Null, Null}, 
   {"C", "I", "26.01", "3P -  7d  1Po", Null, 1128.1713, 0., 1, 3, 
    Null, 1.9601*^6, 0., 0.001122}, 
   {"C", "I", "26.02", "3P -  8d  1Do", Null, 1123.4597, 43.499, 
    5, 5, Null, 63688., 0., 0.000012051}, 
   {"C", "I", "26.02", "3P -  8d  1Do", Null, 1123.119, 
    16.499, 3, 5, Null, 17552., 0., 5.532*^-6}, 
   {"C", "I", "", "3P -  8d  3Fo", Null, 1123.135, 
    43.499, 5, 5, Null, 1214.3, 0., 2.2963*^-7}, 
   {"C", "I", "", "3P -  8d  3Fo", Null, 1123.105, 43.499, 5, 7, 
    Null, 809480., 0., 0.00021439}, 
   {"C", "I", "", "3P -  8d  3Fo", Null, 1122.794, 16.499, 3, 5, 
    Null, 1.7971*^6, 0., 0.00056608}, 
   {"C", "I", "26.03", "3P -  9s  3Po", Null, 1122.649, 
    29.577, 9, 9, 1., 1.18*^7, 0., 0.0022296}, 
   {"C", "I", "26.03", "3P -  9s  3Po", Null, 1123.0654, 
    43.499, 5, 3, 0.13889, 4.9112*^6, 0., 0.0005572}, 
   {"C", "I", "26.03", "3P -  9s  3Po", Null, 1122.7727, 16.499, 
    3, 1, 0.11111, 1.1796*^7, 0., 
    0.000743118}, {"C", "I", "26.03", "3P -  9s  3Po", Null, 
    1122.725, 16.499, 3, 3, 0.08333, 2.9493*^6, 0., 
    0.00055734}, {"C", "I", "26.03", "3P -  9s  3Po", Null, 
    1122.62, 43.499, 5, 5, 0.41667, 
    8.8508*^6, 0., 0.0016723}, 
   {"C", "I", "26.03", "3P -  9s  3Po", Null, 1122.5183, 0., 1, 3, 
    0.11111, 3.9347*^6, 0., 0.0022298}, 
   {"C", "I", "26.03", "3P -  9s  3Po", Null, 1122.28, 
    16.499, 3, 5, 0.13889, 2.9529*^6, 0., 
    0.00092931}, {"C", "I", "27", "3P -  8d  3Do", Null, 
    1122.2933, 29.577, 9, 15, 1., 9.*^6, 0., 0.0028324}\
, {"C", "I", "27", 
      "3P -  8d  3Do", Null, 1122.985, 
    43.499, 5, 3, 0.0055699, 249740., 0., 
    0.00002833}, {"C", "I", "27", "3P -  8d  3Do", Null, 1122.644, 
    16.499, 3, 3, 0.08333, 3.7463*^6, 0., 0.00070786}, 
   {"C", "I", "27", "3P -  8d  3Do", Null, 1122.438, 0., 1, 3, 
    0.11111, 4.9989*^6, 0., 0.002832}, 
   {"C", "I", "27", "3P -  8d  3Do", Null, 1122.3438, 43.499, 5, 
    5, 0.08333, 2.2496*^6, 0., 0.00042483}, 
   {"C", "I", "27", "3P -  8d  3Do", Null, 1122.334, 
    43.499, 5, 7, 0.46667, 8.9991*^6, 0., 
    0.0023792}, {"C", "I", "27", "3P -  8d  3Do", Null, 
    1122.0038, 16.499, 3, 5, 0.25, 6.7552*^6, 0., 
    0.0021249}, {"C", "I", "27.01", "3P -  9s  1Po", Null, 
    1122.2595, 43.499, 5, 3, Null, 1.2498*^8, 0., 
    0.014159}, {"C", "I", "27.01", "3P -  9s  1Po", Null, 
    1121.9196, 16.499, 3, 3, Null, Null, Null, Null}, 
   {"C", "I", "27.01", "3P -  9s  1Po", Null, 1121.7132, 0., 1, 3, 
    Null, Null, Null, Null}, {"C", "I", "27.02", "3P -  8d  1Fo", Null, 
    1122.1795, 43.499, 5, 7, Null, 1.694*^6, 0., 0.00044774}, 
   {"C", "I", "", "3P -  8d  3Po", Null, 1121.8776, 
    29.577, 9, 9, 1., 9.4*^6, 0., 0.0017737}, 
   {"C", "I", "", "3P -  8d  3Po", Null, 1122.098, 43.499, 5, 5, 
    0.41667, 7.0459*^6, 0., 0.00133}, 
   {"C", "I", "", "3P -  8d  3Po", Null, 1121.998, 43.499, 5, 3, 
    0.13889, 3.9154*^6, 0., 0.00044338}, 
   {"C", "I", "", "3P -  8d  3Po", Null, 1121.758, 16.499, 3, 5, 
    0.13889, 2.3508*^6, 0., 0.00073912}, 
   {"C", "I", "", "3P -  8d  3Po", Null, 1121.658, 
    16.499, 3, 3, 0.08333, 2.3513*^6, 0., 0.00044349}, 
   {"C", "I", "", "3P -  8d  3Po", Null, 1121.641, 
    16.499, 3, 1, 0.11111, 9.4059*^6, 0., 
    0.00059135}, {"C", "I", "", "3P -  8d  3Po", Null, 1121.452, 0., 1, 3, 
    0.11111, 3.1369*^6, 0., 0.0017743}, 
   {"C", "I", "27.03", "3P -  8d  1Po", Null, 1122.0657, 43.499, 
    5, 3, Null, 103990., 0., 0.000011777}, 
   {"C", "I", "27.03", "3P -  8d  1Po", Null, 1121.7259, 
    16.499, 3, 3, Null, 1808.2, 0., 3.411*^-7}, 
   {"C", "I", "27.03", "3P -  8d  1Po", Null, 1121.5196, 0., 1, 3, Null, 
    627200., 0., 0.00035481}, 
   {"C", "I", "", "3P -  9d  1Do", Null, 1118.7174, 
    43.499, 5, 5, Null, 344930., 0., 0.000064719}, 
   {"C", "I", "", "3P -  9d  1Do", Null, 1118.3796, 
    16.499, 3, 5, Null, 396270., 0., 
    0.00012385}, {"C", "I", "", "3P -  9d  3Fo", Null, 
    1118.518, 43.499, 5, 5, Null, 49875., 0., 
    9.3547*^-6}, {"C", "I", "", "3P -  9d  3Fo", Null, 
    1118.491, 43.499, 5, 7, Null, 129350., 0., 
    0.000033965}, {"C", "I", "", "3P -  9d  3Fo", Null, 1118.18, 
    16.499, 3, 5, Null, 101890., 0., 
    0.000031833}, {"C", "I", "27.04", "3P - 10s  3Po", Null, 
    1118.0523, 29.577, 9, 9, 1., 1.9*^6, 0., 
    0.00035607}, {"C", "I", "27.04", "3P - 10s  3Po", Null, 
    1118.4629, 43.499, 5, 3, 0.13889, 790800., 0., 
    0.000088985}, {"C", "I", "27.04", "3P - 10s  3Po", Null, 1118.15, 
    16.499, 3, 1, 0.11111, 1.8995*^6, 0., 0.00011868}, 
   {"C", "I", "27.04", "3P - 10s  3Po", Null, 1118.1252, 
    16.499, 3, 3, 0.08333, 474890., 0., 0.000089008}, 
   {"C", "I", "27.04", "3P - 10s  3Po", Null, 1118.03, 
    43.499, 5, 5, 0.41667, 1.4251*^6, 0., 0.00026706}, 
   {"C", "I", "27.04", "3P - 10s  3Po", Null, 1117.9202, 0., 1, 3, 
    0.11111, 633550., 0., 0.00035611}, 
   {"C", "I", "27.04", "3P - 10s  3Po", Null, 1117.69, 16.499, 3, 
    5, 0.13889, 475470., 0., 0.00014841}, 
   {"C", "I", "29", "3P -  9d  3Do", Null, 1117.6901, 29.577, 9, 
    15, 1., 1.8*^7, 0., 0.0056185}, 
   {"C", "I", "29", "3P -  9d  3Do", Null, 1118.408, 
    43.499, 5, 3, 0.0055699, 499440., 0., 
    0.0000561948}, {"C", "I", "29", "3P -  9d  3Do", Null, 
    1118.07, 16.499, 3, 3, 0.08333, 7.4921*^6, 0., 
    0.0014041}, {"C", "I", "29", "3P -  9d  3Do", Null, 1117.866, 
    0., 1, 3, 0.11111, 9.9952*^6, 0., 0.0056176}, 
   {"C", "I", "29", "3P -  9d  3Do", Null, 1117.7302, 43.499, 5, 
    5, 0.08333, 4.4993*^6, 0., 0.00084271}, 
   {"C", "I", "29", "3P -  9d  3Do", Null, 1117.724, 
    43.499, 5, 7, 0.46667, 1.7998*^7, 0., 
    0.0047194}, {"C", "I", "29", "3P -  9d  3Do", Null, 1117.393, 
    16.499, 3, 5, 0.25, 1.3511*^7, 0., 0.0042159}, 
   {"C", "I", "", "3P - 10s  1Po", Null, 1117.674, 
    43.499, 5, 3, Null, 516880., 0., 0.00005808}, 
   {"C", "I", "", "3P - 10s  1Po", Null, 1117.337, 16.499, 3, 3, 
    Null, 259210., 0., 0.000048515}, 
   {"C", "I", "", "3P - 10s  1Po", Null, 1117.132, 0., 1, 3, Null, 410970., 
    0., 0.00023067}, {"C", "I", "", "3P -  9d  1Fo", Null, 
    1117.6215, 43.499, 5, 7, Null, 41923., 0., 
    0.000010991}, {"C", "I", "", "3P -  9d  3Po", Null, 
    1117.581, 43.499, 5, 5, Null, 287490., 0., 
    0.000053831}, {"C", "I", "", "3P -  9d  3Po", Null, 1117.244, 
    16.499, 3, 5, Null, 203650., 0., 0.0000635158}, 
   {"C", "I", "", "3P -  9d  1Po", Null, 1117.542, 
    43.499, 5, 3, Null, 173980., 0., 0.000019545}, 
   {"C", "I", "", "3P -  9d  1Po", Null, 1117.205, 
    16.499, 3, 3, Null, 3808.5, 0., 7.1265*^-7}, 
   {"C", "I", "", "3P -  9d  1Po", Null, 1117.9, 0., 1, 3, Null, 
    693290., 0., 0.00038905}, 
   {"C", "I", "", "3P - 10d  3Fo", Null, 1115.225, 
    43.499, 5, 7, Null, 94912., 0., 
    0.000024776}, {"C", "I", "30", "3P - 10d  3Do", Null, 
    1114.4288, 29.577, 9, 15, 1., 3.6499*^7, 
    0., 0.011296}, {"C", "I", "30", "3P - 10d  3Do", Null, 
    1115.168, 43.499, 5, 3, 0.0055699, 
    1.0099*^6, 0., 0.00011297}, 
   {"C", "I", "30", "3P - 10d  3Do", Null, 1114.832, 16.499, 3, 
    3, 0.08333, 1.5159*^7, 0., 0.0028228}, 
   {"C", "I", "30", "3P - 10d  3Do", Null, 1114.628, 0., 1, 3, 
    0.11111, 2.0211*^7, 0., 0.011294}, 
   {"C", "I", "30", "3P - 10d  3Do", Null, 1114.4611, 43.499, 5, 
    5, 0.08333, 9.0988*^6, 0., 0.0016942}, 
   {"C", "I", "30", "3P - 10d  3Do", Null, 1114.457, 43.499, 5, 
    7, 0.46667, 3.6397*^7, 0., 0.0094882}, 
   {"C", "I", "30", "3P - 10d  3Do", Null, 1114.1258, 16.499, 3, 
    5, 0.25, 2.7322*^7, 0., 0.008474}, 
   {"C", "I", "", "3P - 10d  1Fo", Null, 1114.38, 43.499, 5, 7, 
    Null, 3.7582, 0., 9.7956*^-10}, 
   {"C", "I", "", "3P - 10d  1Po", Null, 1114.332, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 10d  1Po", Null, 
    1113.996, 43.499, 3, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 10d  1Po", Null, 1113.793, 0., 1, 3, Null, 
    7.1353*^6, 0., 0.0039811}, 
   {"C", "I", "", "3P - 11d  3Fo", Null, 1112.825, 
    43.499, 5, 7, Null, 71646., 0., 0.000018622}, 
   {"C", "I", "30.01", "3P - 11d  3Do", Null, 1112.037, 
    29.577, 9, 15, 1., 5.2*^7, 0., 0.016067}, 
   {"C", "I", "30.01", "3P - 11d  3Do", Null, 1112.806, 43.499, 
    5, 3, 0.0055699, 1.4426*^6, 0., 0.00016069}, 
   {"C", "I", "30.01", "3P - 11d  3Do", Null, 1112.472, 
    16.499, 3, 3, 0.08333, 2.164*^7, 0., 0.0040151}, 
   {"C", "I", "30.01", "3P - 11d  3Do", Null, 1112.269, 0., 1, 3, 
    0.11111, 2.8871*^7, 0., 0.016064}, 
   {"C", "I", "30.01", "3P - 11d  3Do", Null, 1112.0593, 
    43.499, 5, 5, 0.08333, 1.2999*^7, 0., 0.00241}, 
   {"C", "I", "30.01", "3P - 11d  3Do", Null, 1112.058, 
    43.499, 5, 7, 0.46667, 5.1997*^7, 0., 
    0.013497}, {"C", "I", "30.01", "3P - 11d  3Do", Null, 
    1111.7255, 16.499, 3, 5, 0.25, 3.9033*^7, 0., 
    0.012054}, {"C", "I", "", "3P - 11d  1Po", Null, 1111.957, 
    43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 11d  1Po", Null, 1111.624, 16.499, 3, 3, 
    Null, 1.928*^7, 0., 0.0035717}, 
   {"C", "I", "", "3P - 11d  1Po", Null, 1111.421, 0., 1, 3, Null, 
    1.6042*^7, 0., 0.0089125}, 
   {"C", "I", "", "3P - 12d  3Do", Null, 1110.441, 0., 1, 3, Null, 
    6.3978*^6, 0., 0.0035481}, 
   {"C", "I", "", "3P - 13s  1Po", Null, 1110.211, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 13s  1Po", Null, 
    1109.878, 16.499, 3, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 13s  1Po", Null, 1109.676, 0., 1, 3, Null, 
    Null, Null, Null}, {"C", "I", "", "3P - 12d  1Po", Null, 1110.169, 
    43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 12d  1Po", Null, 1109.836, 16.499, 3, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 12d  1Po", Null, 1109.634, 
    0., 1, 3, Null, 1.2784*^7, 0., 0.0070795}, 
   {"C", "I", "", "3P - 13d  3Do", Null, 1109.565, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 13d  3Do", Null, 1109.233, 
    16.499, 3, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 13d  3Do", Null, 1109.031, 0., 1, 3, Null, 
    1.6111*^7, 0., 0.0089125}, 
   {"C", "I", "", "3P - 14s  1Po", Null, 1108.816, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 14s  1Po", Null, 1108.484, 
    16.499, 3, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 14s  1Po", Null, 1108.283, 0., 1, 3, Null, 
    Null, Null, Null}, {"C", "I", "", "3P - 13d  1Po", Null, 1108.794, 
    43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 13d  1Po", Null, 1108.462, 
    16.499, 3, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 13d  1Po", Null, 1108.26, 0., 1, 3, Null, Null, 
    Null, Null}, {"C", "I", "", "3P - 14d  3Do", Null, 1108.441, 
    43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 14d  3Do", Null, 1108.109, 16.499, 3, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 14d  3Do", Null, 
    1107.908, 0., 1, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 14d  1Po", Null, 1107.679, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 14d  1Po", Null, 1107.347, 
    16.499, 3, 3, Null, 7.7349*^6, 0., 0.0014219}, 
   {"C", "I", "", "3P - 14d  1Po", Null, 1107.146, 0., 1, 3, Null, 
    2.0826*^6, 0., 0.0011482}, 
   {"C", "I", "", "3P - 15d  3Do", Null, 1107.043, 0., 1, 3, Null, 
    658710., 0., 0.00036308}, {"C", "I", "", "3P - 15d  1Po", Null, 
    1106.781, 43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 15d  1Po", Null, 1106.45, 16.499, 3, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 15d  1Po", Null, 
    1106.25, 0., 1, 3, Null, 830450., 0., 0.00045709}, 
   {"C", "I", "", "3P - 16d  3Do", Null, 1106.315, 0., 1, 3, Null, 
    9.3168*^6, 0., 0.0051286}, 
   {"C", "I", "", "3P - 16d  1Po", Null, 1106.063, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 16d  1Po", Null, 1105.732, 
    16.499, 3, 3, Null, 6.162*^7, 0., 0.011295}, 
   {"C", "I", "", "3P - 16d  1Po", Null, 1105.532, 0., 1, 3, Null, 
    4.1675*^6, 0., 0.0022909}, {"C", "I", "", "3P - 19d  3Fo", Null, 
    1105.329, 43.499, 5, 7, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 17d  1Po", Null, 1105.472, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 17d  1Po", Null, 1105.142, 
    16.499, 3, 3, Null, 1.5495*^7, 0., 0.0028371}, 
   {"C", "I", "", "3P - 17d  1Po", Null, 1104.942, 0., 1, 3, Null, 
    6.6122*^6, 0., 0.0036308}, 
   {"C", "I", "", "3P - 18d  1Po", Null, 1104.957, 43.499, 5, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 18d  1Po", Null, 1104.627, 
    16.499, 3, 3, Null, 4.9045*^6, 0., 0.00089718}, 
   {"C", "I", "", "3P - 18d  1Po", Null, 1104.427, 0., 1, 3, Null, 
    1.3205*^6, 0., 0.00072444}, 
   {"C", "I", "", "3P - 19d  1Po", Null, 1104.52, 
    43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 19d  1Po", Null, 1104.19, 16.499, 3, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 19d  1Po", Null, 1103.991, 
    0., 1, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 20d  1Po", Null, 1104.157, 
    43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 20d  1Po", Null, 1103.828, 16.499, 3, 3, 
    Null, 1.5532*^7, 0., 0.0028371}, 
   {"C", "I", "", "3P - 20d  1Po", Null, 1103.629, 0., 1, 3, Null, 
    2.6386*^7, 0., 0.014454}, {"C", "I", "", "3P - 21d  1Po", Null, 
    1103.845, 43.499, 5, 3, Null, Null, Null, Null}, 
   {"C", "I", "", "3P - 21d  1Po", Null, 1103.516, 16.499, 3, 3, 
    Null, Null, Null, Null}, {"C", "I", "", "3P - 21d  1Po", Null, 
    1103.316, 0., 1, 3, Null, Null, Null, Null}, 
   {"C", "I3", "1Auto", "3P -  2p3 3So", Null, 945.4554, 
    29.577, 9, 3, 1., 6.1*^9, 0., 
    0.27249}, {"C", "I3", "1Auto", "3P -  2p3 3So", Null, 
    945.5798, 43.499, 5, 3, 0.55556, 
    3.3876*^9, 0., 0.27245}, 
   {"C", "I3", "1Auto", "3P -  2p3 3So", Null, 945.338, 
    16.499, 3, 3, 0.33333, 2.0341*^9, 0., 0.27252}, 
   {"C", "I3", "1Auto", "3P -  2p3 3So", Null, 945.191, 0., 1, 3, 
    0.11111, 6.78348*^8, 0., 0.27256}, 
   {"C", "II", "0.01", "2Po-  2p2 4P ", 2328.123, 2328.837, 63.42, 4, 2, 
    Null, 65.599, 120.89, 2.6628*^-8}, 
   {"C", "II", "0.01", "2Po-  2p2 4P ", 2326.931, 2327.645, 63.42, 
    4, 4, Null, 5.24, 6.95, 4.25628*^-9}, 
   {"C", "II", "0.01", "2Po-  2p2 4P ", 2325.3991, 2326.113, 63.42, 
    4, 6, Null, 43.2, 43.2, 5.2565*^-8}, 
   {"C", "II", "0.01", "2Po-  2p2 4P ", 2324.6892, 
    2325.403, 0., 2, 2, Null, 55.3, 120.89, 4.4831*^-8}, 
   {"C", "II", "0.01", "2Po-  2p2 4P ", 2323.5004, 2324.214, 0., 2, 
    4, Null, 1.71, 6.95, 2.7697*^-9}, 
   {"C", "II", "1", "2Po-  2p2 2D ", Null, 1335.3127, 42.289, 6, 
    10, 1., 2.866*^8, 0., 0.12769}, 
   {"C", "II", "1", "2Po-  2p2 2D ", Null, 1335.7077, 63.42, 4, 6, 
    0.6, 2.8635*^8, 2.8635*^8, 0.11488}, 
   {"C", "II", "1", "2Po-  2p2 2D ", Null, 1335.6627, 63.42, 4, 4, 
    0.066679, 4.7732*^7, 2.8698*^8, 0.012766}, 
   {"C", "II", "1", "2Po-  2p2 2D ", Null, 1334.5323, 0., 2, 4, 0.33333, 
    2.3925*^8, 2.8698*^8, 0.12776}, 
   {"C", "II", "2", "2Po-  2p2 2S ", Null, 1036.7909, 42.289, 6, 
    2, 1., 2.29*^9, 0., 0.12301}, 
   {"C", "II", "2", "2Po-  2p2 2S ", Null, 1037.0182, 63.42, 4, 2, 
    0.66667, 1.5257*^9, 2.29*^9, 0.12299}, 
   {"C", "II", "2", "2Po-  2p2 2S ", Null, 1036.3367, 0., 2, 2, 0.33333, 
    7.6433*^8, 2.29*^9, 0.12307}, 
   {"C", "III", "0.01", "1S -  2p  3Po", Null, 1908.734, 0., 1, 3, 1., 
    75.28, 75.28, 1.2322*^-7}, 
   {"C", "III", "1", "1S -  2p  1Po", Null, 977.02, 0., 1, 3, 1., 
    1.7759*^9, 1.7759*^9, 0.76205}, 
   {"C", "IV", "1", "2S -  2p  2Po", Null, 1549.0524, 0., 2, 6, 1., 
    2.6599*^8, 0., 0.28599}, 
   {"C", "IV", "1", "2S -  2p  2Po", Null, 1550.77, 0., 2, 2, 
    0.33333, 2.6412*^8, 2.6412*^8, 0.095225}, 
   {"C", "IV", "1", "2S -  2p  2Po", Null, 1548.195, 0., 2, 4, 
    0.66667, 2.6544*^8, 2.6544*^8, 
    0.19077}, {"N", "I", "1", "4So-  3s  4P ", Null, 1199.9674, 
    0., 4, 12, 1., 4.18*^8, 0., 0.26552}, 
   {"N", "I", "1", "4So-  3s  4P ", Null, 1200.7098, 0., 4, 2, 0.16667, 
    4.0925*^8, 4.0925*^8, 0.0442278}, 
   {"N", "I", "1", "4So-  3s  4P ", Null, 1200.2233, 0., 4, 4, 
    0.33333, 4.0973*^8, 4.0973*^8, 0.088488}, 
   {"N", "I", "1", "4So-  3s  4P ", Null, 1199.5496, 0., 4, 6, 0.5, 
    4.1043*^8, 4.1043*^8, 0.13281}, 
   {"N", "I", "1.01", "4So-  3s  2P ", Null, 1160.937, 0., 4, 2, 
    Null, 23850.9, 7.*^8, 2.4096*^-6}, 
   {"N", "I", "1.01", "4So-  3s  2P ", Null, 1159.817, 0., 4, 4, 
    Null, 42199., 8.*^8, 8.5102*^-6}, 
   {"N", "I", "2", "4So-  2p4 4P ", Null, 1134.6559, 0., 4, 12, 1., 
    1.39*^8, 0., 0.080486}, {"N", "I", "2", "4So-  2p4 4P ", Null, 
    1134.9803, 0., 4, 6, 0.5, 1.3888*^8, 1.3888*^8, 0.040232}, 
   {"N", "I", "2", "4So-  2p4 4P ", Null, 1134.4149, 0., 4, 4, 
    0.33333, 1.3909*^8, 1.3909*^8, 0.026834}, 
   {"N", "I", "2", "4So-  2p4 4P ", Null, 1134.1653, 0., 4, 2, 
    0.16667, 1.3918*^8, 1.3918*^8, 0.01342}, 
   {"N", "I", "2.01", "4So-  3s' 2D ", Null, 1003.377, 0., 4, 6, 
    Null, 1496.5, 5.2798*^8, 3.3889*^-7}, 
   {"N", "I", "2.01", "4So-  3s' 2D ", Null, 1003.372, 0., 4, 4, 
    Null, 323.71, 5.12*^8, 4.8858*^-8}, 
   {"N", "I", "3", "4So-  4s  4P ", Null, 964.3771, 0., 4, 12, 1., 
    7.5628*^7, 2.49*^7, 0.031634}, 
   {"N", "I", "3", "4So-  4s  4P ", Null, 965.0413, 0., 4, 2, 
    0.16231, 7.3599*^7, 9.8499*^7, 0.005131}\
, {"N", "I", "3", 
      "4So-  4s  4P ", Null, 964.6256, 0., 4, 4, 
    0.32923, 7.464*^7, 9.9548*^7, 0.010412}, 
   {"N", "I", "3", "4So-  4s  4P ", Null, 963.9903, 0., 4, 6, 0.50846, 
    7.7*^7, 1.0199*^8, 0.016091}, 
   {"N", "I", "3.01", "4So-  4s  2P ", Null, 960.201, 0., 4, 2, 
    Null, 52286., 1.6799*^8, 3.6136*^-6}, 
   {"N", "I", "3.01", "4So-  4s  2P ", Null, 959.4936, 0., 4, 4, Null, 
    116950., 1.4999*^8, 0.000016141}, 
   {"N", "I", "3.02", "4So-  3d  2P ", Null, 955.8814, 0., 4, 4, 
    Null, 428820., 1.68*^8, 0.000058741}, 
   {"N", "I", "3.02", "4So-  3d  2P ", Null, 955.5292, 0., 4, 2, 
    Null, 252710., 1.6599*^8, 0.000017296}, 
   {"N", "I", "3.03", "4So-  3d  4F ", Null, 955.437, 0., 4, 4, Null, 
    8446.5, 5.528*^7, 1.1569*^-6}, 
   {"N", "I", "3.03", "4So-  3d  4F ", Null, 955.2643, 0., 4, 6, 
    Null, 244760., 4.0599*^7, 0.000050227}, 
   {"N", "I", "3.04", "4So-  3d  2F ", Null, 954.1042, 0., 4, 6, Null, 
    2.1617*^7, 1.88*^8, 0.0044253}, 
   {"N", "I", "3.05", "4So-  3d  4P ", Null, 953.7724, 0., 4, 12, 1., 
    1.6125*^8, 0., 0.065974}, {"N", "I", "3.05", "4So-  3d  4P ", Null, 
    953.9699, 0., 4, 6, 0.50946, 1.642*^8, 1.95*^8, 
    0.033604}, {"N", "I", "3.05", "4So-  3d  4P ", Null, 
    953.6549, 0., 4, 4, 0.32277, 1.562*^8, 
    1.861*^8, 0.021297}, 
   {"N", "I", "3.05", "4So-  3d  4P ", Null, 953.4152, 0., 4, 2, 
    0.16777, 1.625*^8, 1.914*^8, 0.011072}, 
   {"N", "I", "3.06", "4So-  3d  4D ", Null, 952.5227, 0., 4, 2, 
    Null, 6.1286*^6, 3.67*^7, 0.00041681}, 
   {"N", "I", "3.06", "4So-  3d  4D ", Null, 952.4148, 0., 4, 4, Null, 
    1.1449*^7, 4.92*^7, 0.0015558}, 
   {"N", "I", "3.06", "4So-  3d  4D ", Null, 952.3034, 0., 4, 6, 
    Null, 8.6386*^6, 4.51*^7, 0.0017617}, 
   {"N", "I", "3.07", "4So-  3d  2D ", Null, 951.2947, 0., 4, 4, Null, 
    133490., 1.8799*^8, 0.000018111}, 
   {"N", "I", "3.07", "4So-  3d  2D ", Null, 951.0791, 0., 4, 6, 
    Null, 634690., 1.83*^8, 0.0001291}, 
   {"N", "II", "0.01", "3P -  2p3 5So", 2142.773, 2143.448, 130.8, 5, 5, 
    Null, 128., 0., 8.8164*^-8}, 
   {"N", "II", "0.01", "3P -  2p3 5So", 2139.009, 2139.683, 48.7, 3, 5, 
    Null, 57., 0., 6.5205*^-8}, 
   {"N", "II", "1", "3P -  2p3 3Do", Null, 1085.1213, 88.9, 9, 15, 
    1., 3.5*^8, 0., 0.10297}, {"N", "II", "1", "3P -  2p3 3Do", Null, 
    1085.701, 130.8, 5, 7, 0.46667, 3.4944*^8, 
    3.4944*^8, 0.086453}, {"N", "II", "1", "3P -  2p3 3Do", Null, 1085.546, 
    130.8, 5, 5, 0.08333, 8.7394*^7, 3.5029*^8, 0.01544}, 
   {"N", "II", "1", "3P -  2p3 3Do", Null, 1085.529, 130.8, 5, 3, 
    0.0055699, 9.7199*^6, 3.5082*^8, 
    0.0010302}, {"N", "II", "1", "3P -  2p3 3Do", Null, 1084.58, 48.7, 3, 5, 
    0.25, 2.6289*^8, 3.5029*^8, 0.0772698}, 
   {"N", "II", "1", "3P -  2p3 3Do", Null, 1084.562, 48.7, 3, 3, 0.08333, 
    1.4605*^8, 3.5082*^8, 0.025756}, 
   {"N", "II", "1", "3P -  2p3 3Do", Null, 1083.99, 0., 1, 3, 
    0.11111, 1.9505*^8, 3.5082*^8, 0.10308}, 
   {"N", "II", "2", "3P -  2p3 3Po", Null, 916.34658, 88.9, 9, 9, 
    1., 1.1599*^9, 0., 0.14477}, 
   {"N", "II", "2", "3P -  2p3 3Po", Null, 916.71, 130.8, 5, 3, 
    0.13889, 4.7869*^8, 1.1506*^9, 0.036178}\
, {"N", "II", "2", 
      "3P -  2p3 3Po", Null, 916.701, 130.8, 5, 5, 
    0.41667, 8.6151*^8, 1.1493*^9, 
    0.10854}, {"N", "II", "2", "3P -  2p3 3Po", Null, 
    916.02, 48.7, 3, 3, 0.08333, 2.878*^8, 1.1506*^9, 
    0.036204}, {"N", "II", "2", "3P -  2p3 3Po", Null, 916.012, 
    48.7, 3, 5, 0.13889, 2.8782*^8, 1.1493*^9, 
    0.060343}, {"N", "II", "2", "3P -  2p3 3Po", Null, 915.962, 
    48.7, 3, 1, 0.11111, 1.1514*^9, 1.1514*^9, 0.048276}, 
   {"N", "II", "2", "3P -  2p3 3Po", Null, 915.612, 0., 1, 3, 
    0.11111, 3.8425*^8, 1.1506*^9, 0.14488}, 
   {"N", "III", "0.01", "2Po-  2p2 4P ", Null, 1753.995, 174.4, 4, 
    2, Null, 364.99, 703.99, 8.3943*^-8}, 
   {"N", "III", "0.01", "2Po-  2p2 4P ", Null, 1752.16, 174.4, 4, 
    4, Null, 59., 67.959, 2.7155*^-8}, 
   {"N", "III", "0.01", "2Po-  2p2 4P ", Null, 1749.674, 174.4, 4, 
    6, Null, 251.99, 251.99, 1.728*^-7}, 
   {"N", "III", "0.01", "2Po-  2p2 4P ", Null, 1748.646, 0., 2, 2, Null, 
    339.99, 703.99, 1.554*^-7}, 
   {"N", "III", "0.01", "2Po-  2p2 4P ", Null, 1746.823, 0., 2, 4, 
    Null, 8.9599, 67.959, 8.1886*^-9}, 
   {"N", "III", "1", "2Po-  2p2 2D ", Null, 990.9794, 116.266, 6, 
    10, 1., 4.3499*^8, 0., 0.10649}, 
   {"N", "III", "1", "2Po-  2p2 2D ", Null, 991.577, 174.4, 4, 6, 0.6, 
    4.3322*^8, 4.3322*^8, 0.095787}, 
   {"N", "III", "1", "2Po-  2p2 2D ", Null, 991.511, 174.4, 4, 4, 
    0.066679, 7.2221*^7, 4.3518*^8, 0.010644}, 
   {"N", "III", "1", "2Po-  2p2 2D ", Null, 989.799, 0., 2, 4, 0.33333, 
    3.6296*^8, 4.3518*^8, 0.10662}, 
   {"N", "IV", "0.01", "1S -  2p  3Po", Null, 1486.496, 0., 1, 3, 1., 
    577.39, 577.39, 5.7373*^-7}, 
   {"N", "V", "1", "2S -  2p  2Po", Null, 1240.1458, 0., 2, 6, 1., 
    3.4999*^8, 0., 0.23518}, 
   {"N", "V", "1", "2S -  2p  2Po", Null, 1242.804, 0., 2, 2, 0.33333, 
    3.3782*^8, 3.3782*^8, 0.078225}, 
   {"N", "V", "1", "2S -  2p  2Po", Null, 1238.821, 0., 2, 4, 
    0.66667, 3.4109*^8, 3.4109*^8, 0.15696}, 
   {"O", "I", "1", "3P -  3s  5So", Null, 1358.5123, 158.265, 3, 
    5, Null, 1350., 0., 6.22548*^-7}, 
   {"O", "I", "1", "3P -  3s  5So", Null, 1355.5977, 0., 5, 5, Null, 
    4530.99, 0., 1.2489*^-6}, 
   {"O", "I", "2", "3P -  3s  3So", Null, 1303.492, 77.974, 9, 3, 
    1., 5.75*^8, 0., 0.048822}, 
   {"O", "I", "2", "3P -  3s  3So", Null, 1306.0286, 226.977, 1, 3, 
    0.11111, 6.3517*^7, 5.75*^8, 0.048727}, 
   {"O", "I", "2", "3P -  3s  3So", Null, 1304.8576, 158.265, 3, 
    3, 0.33333, 1.9106*^8, 5.75*^8, 0.048771}, 
   {"O", "I", "2", "3P -  3s  3So", Null, 1302.1685, 0., 5, 3, 0.55556, 
    3.2042*^8, 5.75*^8, 0.048873}, 
   {"O", "I", "2.01", "3P -  4s  5So", Null, 1049.115, 158.265, 3, 
    5, Null, Null, Null, Null}, 
   {"O", "I", "2.01", "3P -  4s  5So", Null, 1047.376, 0., 5, 5, Null, Null, 
    Null, Null}, {"O", "I", "3", "3P -  4s  3So", Null, 1040.0733, 
    77.974, 9, 3, 1., 1.7*^8, 1.88*^7, 0.0091899}, 
   {"O", "I", "3", "3P -  4s  3So", Null, 1041.6876, 226.977, 1, 3, 
    0.11111, 1.8801*^7, 1.888*^8, 0.0091756}\
, {"O", "I", "3", 
      "3P -  4s  3So", Null, 1040.9425, 158.265, 3, 3, 
    0.33333, 5.6524*^7, 1.888*^8, 0.0091822}, 
   {"O", "I", "3", "3P -  4s  3So", Null, 1039.2304, 0., 5, 3, 
    0.55556, 9.4675*^7, 1.888*^8, 0.00919758}, 
   {"O", "I", "3.01", "3P -  3d  5Do", Null, 1028.8705, 226.977, 1, 3, Null, 
    982430., 4.3514*^7, 0.00046774}, 
   {"O", "I", "3.01", "3P -  3d  5Do", Null, 1028.1447, 
    158.265, 3, 5, Null, 3.9*^6, 4.6982*^7, 0.0010301}, 
   {"O", "I", "3.01", "3P -  3d  5Do", Null, 1028.1436, 158.265, 
    3, 3, Null, 606620., 4.3514*^7, 0.000096134}, 
   {"O", "I", "3.01", "3P -  3d  5Do", Null, 1028.1431, 158.265, 
    3, 1, Null, 9333.3, 4.1909*^7, 4.93048*^-7}, 
   {"O", "I", "3.01", "3P -  3d  5Do", Null, 1026.4757, 0., 5, 7, 
    Null, 1.1126*^7, 5.3026*^7, 0.0024605}, 
   {"O", "I", "3.01", "3P -  3d  5Do", Null, 1026.4744, 0., 5, 5, Null, 
    1.1816*^6, 4.6982*^7, 0.00018665}, 
   {"O", "I", "3.01", "3P -  3d  5Do", Null, 1026.4733, 0., 5, 3, Null, 
    24793., 4.3514*^7, 2.3498*^-6}, 
   {"O", "I", "4", "3P -  3d  3Do", Null, 1026.5834, 77.974, 9, 15, 1., 
    7.7*^7, 2.35*^7, 0.020276}, 
   {"O", "I", "4", "3P -  3d  3Do", Null, 1028.1571, 226.977, 1, 3, 
    0.11111, 4.2581*^7, 1.0023*^8, 0.020245}, 
   {"O", "I", "4", "3P -  3d  3Do", Null, 1027.4313, 
    158.265, 3, 3, 0.08333, 3.2003*^7, 
    1.0023*^8, 0.0050646}, 
   {"O", "I", "4", "3P -  3d  3Do", Null, 1027.4305, 
    158.265, 3, 5, 0.25, 5.7607*^7, 1.004*^8, 0.015195}, 
   {"O", "I", "4", "3P -  3d  3Do", Null, 1025.7633, 0., 5, 3, 
    0.0055699, 2.1457*^6, 1.0023*^8, 
    0.00020309}, {"O", "I", "4", "3P -  3d  3Do", Null, 
    1025.7626, 0., 5, 5, 0.08333, 1.9295*^7, 1.004*^8, 
    0.0030437}, {"O", "I", "4", "3P -  3d  3Do", Null, 1025.7616, 
    0., 5, 7, 0.46667, 7.7186*^7, 1.0069*^8, 0.017046}, 
   {"O", "I", "5", "3P -  3s' 3Do", Null, 989.45768, 77.974, 9, 15, 
    1., 2.1*^8, 2.9*^7, 0.051371}, 
   {"O", "I", "5", "3P -  3s' 3Do", Null, 990.801, 226.977, 1, 3, 
    0.11111, 1.1619*^8, 2.3836*^8, 0.051301}\
, {"O", "I", "5", 
      "3P -  3s' 3Do", Null, 990.2043, 158.265, 3, 5, 
    0.25, 1.5714*^8, 2.3877*^8, 0.038499}, 
   {"O", "I", "5", "3P -  3s' 3Do", Null, 990.1269, 
    158.265, 3, 3, 0.08333, 8.7319*^7, 
    2.3836*^8, 0.012834}, 
   {"O", "I", "5", "3P -  3s' 3Do", Null, 988.7734, 0., 5, 7, 
    0.46667, 2.1044*^8, 2.3944*^8, 0.043182}, 
   {"O", "I", "5", "3P -  3s' 3Do", Null, 988.65498, 0., 5, 5, 
    0.08333, 5.2626*^7, 2.3877*^8, 0.0077116}, 
   {"O", "I", "5", "3P -  3s' 3Do", Null, 988.5778, 0., 5, 3, 
    0.0055699, 5.8536*^6, 2.3836*^8, 0.00051458}, 
   {"O", "I", "6", "3P -  5s  5So", Null, 980.792, 
    158.265, 3, 5, Null, Null, Null, Null}, 
   {"O", "I", "6", "3P -  5s  5So", Null, 979.272, 0., 5, 5, Null, 
    Null, Null, Null}, {"O", "I", "7", "3P -  5s  3So", Null, 977.1921, 
    77.974, 9, 3, 1., 6.91*^7, 1.1*^7, 0.0032974}, 
   {"O", "I", "7", "3P -  5s  3So", Null, 978.617, 226.977, 1, 3, 
    0.11111, 7.6442*^6, 8.01*^7, 
    0.0032926}, {"O", "I", "7", "3P -  5s  3So", Null, 977.9594, 
    158.265, 3, 3, 0.33333, 2.2979*^7, 
    8.01*^7, 0.0032948}, 
   {"O", "I", "7", "3P -  5s  3So", Null, 976.4481, 0., 5, 3, 
    0.55556, 3.8477*^7, 8.01*^7, 0.0033}, 
   {"O", "I", "8", "3P -  3s' 1Do", Null, 975.574, 
    158.265, 3, 5, Null, Null, Null, Null}, 
   {"O", "I", "8", "3P -  3s' 1Do", Null, 974.07, 0., 5, 5, Null, 
    140599.9999999999, 0., 0.00002}, 
   {"O", "I", "9", "3P -  4d  5Do", Null, 974.2916, 226.977, 1, 3, 
    Null, 208760., 1.3783*^7, 0.000089125}, 
   {"O", "I", "9", "3P -  4d  5Do", Null, 973.6402, 
    158.265, 3, 5, Null, 738540., 1.4397*^7, 
    0.00017494}, {"O", "I", "9", "3P -  4d  5Do", Null, 
    973.6398, 158.265, 3, 3, Null, 128890., 
    1.3783*^7, 0.000018318}, {"O", "I", "9", "3P -  4d  5Do", Null, 
    973.6395, 158.265, 3, 1, Null, 1983.1, 
    1.3442*^7, 9.3946*^-8}, 
   {"O", "I", "9", "3P -  4d  5Do", Null, 972.1428, 0., 5, 7, Null, 
    1.598*^6, 1.5038*^7, 0.00031698}, 
   {"O", "I", "9", "3P -  4d  5Do", Null, 972.1422, 0., 5, 5, Null, 218630., 
    1.4397*^7, 0.000030976}, 
   {"O", "I", "9", "3P -  4d  5Do", Null, 972.1418, 0., 5, 3, Null, 
    5267., 1.3783*^7, 4.4774*^-7}, 
   {"O", "I", "10", "3P -  4d  3Do", Null, 972.4746, 77.974, 9, 15, 
    1., 6.2699*^7, 3.25*^6, 0.014792}, 
   {"O", "I", "10", "3P -  4d  3Do", Null, 973.8852, 226.977, 1, 3, 
    0.11111, 3.4627*^7, 6.5642*^7, 0.014771}\
, {"O", "I", "10", 
      "3P -  4d  3Do", Null, 973.2343, 158.265, 3, 5, 
    0.25, 4.6849*^7, 6.5775*^7, 0.011086}, 
   {"O", "I", "10", "3P -  4d  3Do", Null, 973.2339, 
    158.265, 3, 3, 0.08333, 2.6021*^7, 
    6.5642*^7, 0.0036959}, 
   {"O", "I", "10", "3P -  4d  3Do", Null, 971.7382, 0., 5, 7, 
    0.46667, 6.2743*^7, 6.5993*^7, 0.012435}\
, {"O", "I", "10", 
      "3P -  4d  3Do", Null, 971.7376, 0., 5, 5, 
    0.08333, 1.5685*^7, 6.5775*^7, 0.0022204}, 
   {"O", "I", "10", "3P -  4d  3Do", Null, 971.7371, 0., 5, 3, 
    0.0055699, 1.7442*^6, 6.5642*^7, 0.00014815}, 
   {"O", "I", "", "3P -  6s  5So", Null, 953.643, 
    158.265, 3, 5, Null, Null, Null, Null}, 
   {"O", "I", "", "3P -  6s  5So", Null, 952.2059, 0., 5, 5, Null, Null, 
    Null, Null}, {"O", "I", "11", "3P -  6s  3So", Null, 951.5901, 
    77.974, 9, 3, 1., 3.47*^7, 4.0699*^6, 0.0015702}, 
   {"O", "I", "11", "3P -  6s  3So", Null, 952.9413, 226.977, 1, 3, 
    0.11111, 3.8391*^6, 3.876*^7, 0.001568}, 
   {"O", "I", "11", "3P -  6s  3So", Null, 952.3178, 
    158.265, 3, 3, 0.33333, 1.154*^7, 3.876*^7, 
    0.0015699}, {"O", "I", "11", "3P -  6s  3So", Null, 950.8846, 
    0., 5, 3, 0.55556, 1.9321*^7, 3.876*^7, 0.0015714}, 
   {"O", "I", "12", "3P -  5d  3Do", Null, 949.38788, 77.974, 9, 15, 
    1., 2.86*^7, 0., 0.0064411}, 
   {"O", "I", "12", "3P -  5d  3Do", Null, 950.7327, 226.977, 1, 3, 
    0.11111, 1.5821*^7, 0., 0.0064319}, 
   {"O", "I", "12", "3P -  5d  3Do", Null, 950.1121, 
    158.265, 3, 5, 0.25, 2.1401*^7, 0., 0.0048271}, 
   {"O", "I", "12", "3P -  5d  3Do", Null, 950.1121, 
    158.265, 3, 3, 0.08333, 1.1889*^7, 0., 
    0.0016099}, {"O", "I", "12", "3P -  5d  3Do", Null, 948.6855, 
    0., 5, 7, 0.46667, 2.8664*^7, 0., 0.0054146}, 
   {"O", "I", "12", "3P -  5d  3Do", Null, 948.6855, 0., 5, 5, 0.08333, 
    7.1656*^6, 0., 0.00096684}, 
   {"O", "I", "12", "3P -  5d  3Do", Null, 948.6855, 0., 5, 3, 
    0.0055699, 796850., 0., 0.00006451}, 
   {"O", "I", "13", "3P -  7s  3So", Null, 938.5269, 77.974, 9, 3, 
    1., 1.98*^7, 0., 0.00087155}, 
   {"O", "I", "13", "3P -  7s  3So", Null, 939.84128, 226.977, 1, 3, 
    0.11111, 2.1908*^6, 0., 0.00087033}, 
   {"O", "I", "13", "3P -  7s  3So", Null, 939.2346, 
    158.265, 3, 3, 0.33333, 6.5859*^6, 0., 
    0.00087089}, {"O", "I", "13", "3P -  7s  3So", Null, 
    937.8405, 0., 5, 3, 0.55556, 1.1024*^7, 0., 
    0.00087227}, {"O", "I", "14", "3P -  6d  3Do", Null, 
    937.314, 77.974, 9, 15, 1., 1.7*^7, 0., 0.0037319}, 
   {"O", "I", "14", "3P -  6d  3Do", Null, 938.6249, 226.977, 1, 3, 
    0.11111, 9.4048*^6, 0., 0.0037266}, 
   {"O", "I", "14", "3P -  6d  3Do", Null, 938.028, 
    158.265, 3, 5, 0.25, 1.2721*^7, 0., 0.0027968}, 
   {"O", "I", "14", "3P -  6d  3Do", Null, 938.028, 
    158.265, 3, 3, 0.08333, 7.0671*^6, 0., 0.00093222}, 
   {"O", "I", "14", "3P -  6d  3Do", Null, 936.6295, 0., 5, 7, 
    0.46667, 1.7037*^7, 0., 0.0031371}, 
   {"O", "I", "14", "3P -  6d  3Do", Null, 936.6295, 0., 5, 5, 
    0.08333, 4.2592*^6, 0., 0.00056016}, 
   {"O", "I", "14", "3P -  6d  3Do", Null, 936.6295, 0., 5, 3, 
    0.0055699, 473640., 0., 0.000037376}, 
   {"O", "I", "15", "3P -  8s  3So", Null, 930.9318, 77.974, 9, 3, 
    1., 1.2499*^7, 0., 0.00053702}, 
   {"O", "I", "15", "3P -  8s  3So", Null, 932.2249, 226.977, 1, 3, 
    0.11111, 1.372*^6, 0., 0.00053627}, 
   {"O", "I", "15", "3P -  8s  3So", Null, 931.6282, 
    158.265, 3, 3, 0.33333, 4.1249*^6, 0., 0.00053662}, 
   {"O", "I", "15", "3P -  8s  3So", Null, 930.2566, 0., 5, 3, 0.55556, 
    6.904*^6, 0., 0.00053742}, {"O", "I", "16", "3P -  7d  3Do", Null, 
    930.191, 77.974, 9, 15, 1., 1.09*^7, 0., 0.0023565}, 
   {"O", "I", "16", "3P -  7d  3Do", Null, 931.482, 226.977, 1, 3, 
    0.11111, 6.0304*^6, 0., 0.0023533}, 
   {"O", "I", "16", "3P -  7d  3Do", Null, 930.8862, 
    158.265, 3, 5, 0.25, 8.1567*^6, 0., 0.0017661}, 
   {"O", "I", "16", "3P -  7d  3Do", Null, 930.8862, 
    158.265, 3, 3, 0.08333, 4.5313*^6, 0., 0.00058867}, 
   {"O", "I", "16", "3P -  7d  3Do", Null, 929.5168, 0., 5, 7, 
    0.46667, 1.0924*^7, 0., 0.0019809}, 
   {"O", "I", "16", "3P -  7d  3Do", Null, 929.5168, 0., 5, 5, 0.08333, 
    2.7308*^6, 0., 0.00035372}, 
   {"O", "I", "16", "3P -  7d  3Do", Null, 929.5168, 0., 5, 3, 
    0.0055699, 303680., 0., 0.000023601}, 
   {"O", "I", "17", "3P -  9s  3So", Null, 926.1143, 77.974, 9, 3, 1., 
    8.25*^6, 0., 0.00035361}, 
   {"O", "I", "17", "3P -  9s  3So", Null, 927.394, 226.977, 1, 3, 
    0.11111, 912870., 0., 0.00035311}, 
   {"O", "I", "17", "3P -  9s  3So", Null, 926.809, 
    158.265, 3, 3, 0.33333, 2.7438*^6, 0., 
    0.00035334}, {"O", "I", "17", "3P -  9s  3So", Null, 
    925.442, 0., 5, 3, 0.55556, 4.5934*^6, 0., 
    0.00035386}, {"O", "I", "18", "3P -  8d  3Do", Null, 925.6163, 77.974, 
    9, 15, 1., 7.4399*^6, 0., 0.0015906}, 
   {"O", "I", "18", "3P -  8d  3Do", Null, 926.903, 226.977, 1, 3, 
    0.11111, 4.1106*^6, 0., 0.0015884}, 
   {"O", "I", "18", "3P -  8d  3Do", Null, 926.295, 158.265, 3, 5, 
    0.25, 5.5603*^6, 0., 0.0011921}, 
   {"O", "I", "18", "3P -  8d  3Do", Null, 926.295, 158.265, 3, 3, 
    0.08333, 3.0889*^6, 0., 0.00039734}, 
   {"O", "I", "18", "3P -  8d  3Do", Null, 924.952, 0., 5, 7, 
    0.46667, 7.4461*^6, 0., 0.0013371}, 
   {"O", "I", "18", "3P -  8d  3Do", Null, 924.952, 0., 5, 5, 0.08333, 
    1.8614*^6, 0., 0.00023875}, 
   {"O", "I", "18", "3P -  8d  3Do", Null, 924.952, 0., 5, 3, 
    0.0055699, 206999.9999999999, 0., 0.000015939}, 
   {"O", "I", "19", "3P - 10s  3So", Null, 922.86428, 77.974, 9, 3, 
    1., 5.75*^6, 0., 0.00024473}, 
   {"O", "I", "19", "3P - 10s  3So", Null, 924.135, 226.977, 1, 3, 
    0.11111, 636250., 0., 0.00024439}, 
   {"O", "I", "19", "3P - 10s  3So", Null, 923.549, 158.265, 3, 3, 
    0.33333, 1.9124*^6, 0., 0.00024454}, 
   {"O", "I", "19", "3P - 10s  3So", Null, 922.2, 0., 5, 3, 
    0.55556, 3.2014*^6, 0., 0.00024499}, 
   {"O", "I", "20", "3P -  9d  3Do", Null, 922.5202, 77.974, 9, 15, 
    1., 5.2699*^6, 0., 0.0011185}, 
   {"O", "I", "20", "3P -  9d  3Do", Null, 923.79, 226.977, 1, 3, 
    0.11111, 2.9102*^6, 0., 0.0011179}, 
   {"O", "I", "20", "3P -  9d  3Do", Null, 923.2, 
    158.265, 3, 5, 0.25, 3.9363*^6, 0., 0.00083827}, 
   {"O", "I", "20", "3P -  9d  3Do", Null, 923.2, 
    158.265, 3, 3, 0.08333, 2.1867*^6, 0., 0.00027941}, 
   {"O", "I", "20", "3P -  9d  3Do", Null, 921.86, 0., 5, 7, 
    0.46667, 5.2713*^6, 0., 0.00094023}, 
   {"O", "I", "20", "3P -  9d  3Do", Null, 921.86, 0., 5, 5, 0.08333, 
    1.3178*^6, 0., 0.00016789}, 
   {"O", "I", "20", "3P -  9d  3Do", Null, 921.86, 0., 5, 3, 
    0.0055699, 146540., 0., 0.000011202}, 
   {"O", "I", "21", "3P - 11s  3So", Null, 920.5955, 77.974, 9, 3, 
    1., 4.2*^6, 0., 0.00017788}, 
   {"O", "I", "21", "3P - 11s  3So", Null, 921.86, 226.977, 1, 3, 
    0.11111, 464740., 0., 0.00017763}, 
   {"O", "I", "21", "3P - 11s  3So", Null, 921.247, 
    158.265, 3, 3, 0.33333, 1.397*^6, 0., 0.00017775}, 
   {"O", "I", "21", "3P - 11s  3So", Null, 919.908, 0., 5, 3, 0.55556, 
    2.3386*^6, 0., 0.00017801}, 
   {"O", "I", "22", "3P - 10d  3Do", Null, 920.3188, 77.974, 9, 15, 1., 
    3.8499*^6, 0., 0.00081267}, 
   {"O", "I", "22", "3P - 10d  3Do", Null, 921.575, 226.977, 1, 3, 
    0.11111, 2.1246*^6, 0., 0.00081155}, 
   {"O", "I", "22", "3P - 10d  3Do", Null, 921.0058, 
    158.265, 3, 5, 0.25, 2.8736*^6, 0., 0.00060905}, 
   {"O", "I", "22", "3P - 10d  3Do", Null, 921.0058, 
    158.265, 3, 3, 0.08333, 1.5964*^6, 0., 0.00020301}, 
   {"O", "I", "22", "3P - 10d  3Do", Null, 919.658, 0., 5, 7, 
    0.46667, 3.8483*^6, 0., 0.00068314}, 
   {"O", "I", "22", "3P - 10d  3Do", Null, 919.658, 0., 5, 5, 0.08333, 
    962030., 0., 0.00012198}, 
   {"O", "I", "22", "3P - 10d  3Do", Null, 919.658, 0., 5, 3, 
    0.0055699, 106980., 0., 8.1399*^-6}, 
   {"O", "I", "24", "3P - 11d  3Do", Null, 918.7017, 77.974, 9, 15, 1., 
    2.91*^6, 0., 0.00061369}, 
   {"O", "I", "24", "3P - 11d  3Do", Null, 919.961, 226.977, 1, 3, 
    0.11111, 1.61*^6, 0., 0.00061284}, 
   {"O", "I", "24", "3P - 11d  3Do", Null, 919.38, 
    158.265, 3, 5, 0.25, 2.1777*^6, 0., 0.00045993}, 
   {"O", "I", "24", "3P - 11d  3Do", Null, 919.38, 
    158.265, 3, 3, 0.08333, 1.2098*^6, 0., 
    0.00015339}, {"O", "I", "24", "3P - 11d  3Do", Null, 918.044, 
    0., 5, 7, 0.46667, 2.9163*^6, 0., 0.00051587}, 
   {"O", "I", "24", "3P - 11d  3Do", Null, 918.044, 0., 5, 5, 0.08333, 
    729040., 0., 0.000092115}, 
   {"O", "I", "24", "3P - 11d  3Do", Null, 918.044, 0., 5, 3, 
    0.0055699, 81072., 0., 6.1462*^-6}, 
   {"O", "I", "26", "3P - 12d  3Do", Null, 917.4706, 77.974, 9, 15, 1., 
    2.25*^6, 0., 0.00047323}, {"O", "I", "26", "3P - 12d  3Do", Null, 
    918.726, 226.977, 1, 3, 0.11111, 1.2449*^6, 0., 
    0.00047258}, {"O", "I", "26", "3P - 12d  3Do", Null, 918.147, 
    158.265, 3, 5, 0.25, 1.6838*^6, 0., 0.00035466}, 
   {"O", "I", "26", "3P - 12d  3Do", Null, 918.147, 158.265, 3, 3, 
    0.08333, 935390., 0., 0.00011822}, 
   {"O", "I", "26", "3P - 12d  3Do", Null, 916.815, 0., 5, 7, 
    0.46667, 2.2548*^6, 0., 0.0003978}, 
   {"O", "I", "26", "3P - 12d  3Do", Null, 916.815, 0., 5, 5, 0.08333, 
    563690., 0., 0.000071032}, 
   {"O", "I", "26", "3P - 12d  3Do", Null, 916.815, 0., 5, 3, 
    0.0055699, 62684.8, 0., 4.7395*^-6}, 
   {"O", "I", "28", "3P - 13d  3Do", Null, 917.99, 0., 9, 15, 1., 
    1.77*^6, 0., 0.00037189}, {"O", "III", "0.01", "3P -  2p3 5So", Null, 
    1666.15, 306.174, 5, 5, Null, 584.45, 0., 
    2.4324*^-7}, {"O", "III", "0.01", "3P -  2p3 5So", Null, 
    1660.809, 113.178, 3, 5, Null, 236.88, 0., 
    1.6326*^-7}, {"O", "IV", "0.01", "2Po-  2p2 4P ", Null, 1407.382, 
    385.99, 4, 2, Null, 2119.8, 4194.7, 
    3.1473*^-7}, {"O", "IV", "0.01", "2Po-  2p2 4P ", Null, 
    1404.806, 385.99, 4, 4, Null, 441.42, 499.57, 
    1.306*^-7}, {"O", "IV", "0.01", "2Po-  2p2 4P ", Null, 
    1401.157, 385.99, 4, 6, Null, 1465.6, 1465.6, 
    6.4705*^-7}, {"O", "IV", "0.01", "2Po-  2p2 4P ", Null, 1399.78, 0., 2, 
    2, Null, 2074.9, 4194.7, 6.0949*^-7}, 
   {"O", "IV", "0.01", "2Po-  2p2 4P ", Null, 1397.232, 0., 2, 4, Null, 
    58.15, 499.57, 3.4039*^-8}, 
   {"O", "V", "0.01", "1S -  2p  3Po", Null, 1218.344, 0., 1, 3, 1., 
    2210.99, 2210.99, 1.4754*^-6}, 
   {"O", "VI", "1", "2S -  2p  2Po", Null, 1033.816, 0., 2, 6, 1., 
    4.1498*^8, 0., 0.19901}, 
   {"O", "VI", "1", "2S -  2p  2Po", Null, 1037.6167, 0., 2, 2, 0.33333, 
    4.0946*^8, 4.0946*^8, 0.066091}, 
   {"O", "VI", "1", "2S -  2p  2Po", Null, 1031.9261, 0., 2, 4, 
    0.66667, 4.1628*^8, 4.1628*^8, 
    0.13291},  
   {"F", "I", "", "2Po-  3s  4P ", Null, 977.743, 404.1, 2, 4, Null, 
    296230., 2.3619*^6, 0.000084912}, 
   {"F", "I", "", "2Po-  3s  4P ", Null, 976.508, 0., 4, 6, Null, 
    29967., 29967., 6.4269*^-6}, 
   {"F", "I", "", "2Po-  3s  4P ", Null, 976.217, 404.1, 2, 2, 
    Null, 580790., 734760., 0.000082979}, 
   {"F", "I", "", "2Po-  3s  4P ", Null, 973.895, 0., 4, 4, Null, 
    2.0657*^6, 2.3619*^6, 0.00029372}, 
   {"F", "I", "", "2Po-  3s  4P ", Null, 972.38, 0., 4, 2, Null, 153970., 
    734760., 0.000010913}, 
   {"F", "I", "1", "2Po-  3s  2P ", Null, 955.0663, 134.79, 6, 6, 
    1., 7.6999*^8, 0., 0.10393}, 
   {"F", "I", "1", "2Po-  3s  2P ", Null, 958.5258, 404.1, 2, 4, 
    0.11111, 1.2539*^8, 7.5912*^8, 0.034518}\
, {"F", "I", "1", 
        "2Po-  3s  2P ", Null, 955.546, 404.1, 2, 2, 
    0.22222, 5.0599*^8, 7.6179*^8, 0.069251}\
, {"F", "I", "1", 
      "2Po-  3s  2P ", Null, 954.826, 0., 4, 4, 0.55556, 
    6.33828*^8, 7.5912*^8, 0.08663}, 
   {"F", "I", "1", "2Po-  3s  2P ", Null, 951.87, 0., 4, 2, 
    0.11111, 2.5589*^8, 7.6179*^8, 0.017389}, 
   {"F", "IV", "", "3P -  2p3 5So", Null, 1359.052, 
    614.99, 5, 5, Null, 48832., 0., 0.000013522}, 
   {"F", "IV", "", "3P -  2p3 5So", Null, 1351.923, 226., 3, 5, 
    Null, 1021.6, 0., 4.6653*^-7}, 
   {"F", "V", "", "2Po-  2p2 4P ", Null, 1175.838, 744.5, 4, 2, 
    Null, 4630.99, 9390., 4.7985*^-7}, 
   {"F", "V", "", "2Po-  2p2 4P ", Null, 1172.347, 744.5, 4, 4, Null, 
    979.99, 1109.9, 2.0172*^-7}, 
   {"F", "V", "", "2Po-  2p2 4P ", Null, 1167.372, 744.5, 4, 6, 
    Null, 3720., 3720., 1.1499*^-6}, 
   {"F", "V", "", "2Po-  2p2 4P ", Null, 1165.634, 0., 2, 2, Null, 
    4760.99, 9390., 9.6959*^-7}, 
   {"F", "V", "", "2Po-  2p2 4P ", Null, 1162.203, 0., 2, 4, Null, 
    130., 1109.9, 5.2649*^-8}, 
   {"F", "VI", "", "1S -  2p  3Po", Null, 1032.52, 0., 1, 3, 1., 
    7320., 7320., 3.5098*^-6}, 
   {"Ne", "V", "", "3P -  2p3 5So", Null, 1146.1, 
    112.99, 5, 5, Null, 5113.6, 0., 
    1.0079*^-6}, {"Ne", "V", "", "3P -  2p3 5So", Null, 1137., 
    414.99, 3, 5, Null, 1801.6, 0., 5.8194*^-7}, 
   {"Ne", "VI", "", "2Po-  2p2 4P ", Null, 1010.6, 316., 4, 2, 
    Null, Null, Null, Null}, {"Ne", "VI", "", "2Po-  2p2 4P ", Null, 
    1006.1, 316., 4, 4, Null, Null, Null, Null}, 
   {"Ne", "VI", "", "2Po-  2p2 4P ", Null, 999.6, 316., 4, 6, Null, Null, 
    Null, Null}, {"Ne", "VI", "", "2Po-  2p2 4P ", Null, 997.4, 0., 2, 2, 
    Null, Null, Null, Null}, {"Ne", "VI", "", "2Po-  2p2 4P ", Null, 
    993.99, 0., 2, 4, Null, Null, Null, Null}, 
   {"Na", "I", "1v", "2S -  3p  2Po", 5891.9404, 5893.5736, 
    0., 2, 6, 1., 6.075*^7, 0., 0.94903}, 
   {"Na", "I", "1v", "2S -  3p  2Po", 5895.924, 5897.5581, 
    0., 2, 2, 0.33333, 6.0627*^7, 6.0627*^7, 0.31613}, 
   {"Na", "I", "1v", "2S -  3p  2Po", 5889.9511, 5891.5833, 0., 2, 
    4, 0.66667, 6.0812*^7, 6.0812*^7, 
    0.6329}, {"Na", "I", "1v", "2S -  3p  2Po", 5891.9404, 
    5893.5736, 0., 2, 6, 1., 6.0753*^7, 0., 0.94908}, 
   {"Na", "I", "1v", "2S -  3p  2Po", 5895.924, 5897.5581, 
    0., 2, 2, 0.33524, 6.0976*^7, 
    6.0976*^7, 0.31795}, 
   {"Na", "I", "1v", "2S -  3p  2Po", 5889.9511, 5891.5833, 0., 2, 
    4, 0.66476, 6.0641*^7, 6.0641*^7, 
    0.63113}, {"Na", "I", "2v", "2S -  4p  2Po", 3302.5712, 
    3303.5223, 0., 2, 6, 1., 2.7499*^6, 6.9*^6, 
    0.013449}, {"Na", "I", "2v", "2S -  4p  2Po", 3302.978, 
    3303.929, 0., 2, 2, 0.33333, 2.7399*^6, 9.6399*^6, 
    0.0044823}, {"Na", "I", "2v", "2S -  4p  2Po", 3302.3681, 
    3303.319, 0., 2, 4, 0.66667, 2.7405*^6, 
    9.6405*^6, 0.0089665}, 
   {"Na", "I", "1", "2S -  5p  2Po", 2852.878, 2853.716, 0., 2, 6, 1., 
    520000., 0., 0.0019046}, {"Na", "I", "1", "2S -  5p  2Po", 2853.012, 
    2853.85, 0., 2, 2, 0.33333, 519920., 0., 0.00063483}, 
   {"Na", "I", "1", "2S -  5p  2Po", 2852.811, 2853.649, 0., 2, 4, 
    0.66667, 520040., 0., 0.0012698}, 
   {"Na", "I", "2", "2S -  6p  2Po", 2680.372, 2681.168, 0., 2, 6, 
    1., 225999.9999999999, 0., 0.00073069}, 
   {"Na", "I", "2", "2S -  6p  2Po", 2680.434, 2681.23, 0., 2, 2, 0.33333, 
    225980., 0., 0.00024356}, 
   {"Na", "I", "2", "2S -  6p  2Po", 2680.341, 2681.137, 0., 2, 4, 
    0.66667, 226010., 0., 0.00048714}, 
   {"Na", "I", "3", "2S -  7p  2Po", 2593.8852, 2594.661, 0., 2, 6, 
    1., 119999.9999999999, 0., 0.00036335}, 
   {"Na", "I", "3", "2S -  7p  2Po", 2593.9194, 2594.695, 0., 2, 2, 0.33333, 
    119990., 0., 0.00012111}, 
   {"Na", "I", "3", "2S -  7p  2Po", 2593.8684, 2594.644, 0., 2, 4, 
    0.66667, 119999.9999999999, 0., 0.00024223}, 
   {"Na", "I", "4", "2S -  8p  2Po", 2543.851, 2544.6147, 0., 2, 6, 1., 
    65999.99999999999, 0., 0.00019221}, 
   {"Na", "I", "4", "2S -  8p  2Po", 2543.872, 2544.636, 
    0., 2, 2, 0.33333, 65998., 0., 0.000064067}, 
   {"Na", "I", "4", "2S -  8p  2Po", 2543.84, 2544.604, 0., 2, 4, 
    0.66667, 66001., 0., 0.00012814}, 
   {"Na", "I", "5", "2S -  9p  2Po", 2512.1474, 2512.904, 
    0., 2, 6, 1., 40499.99999999999, 0., 0.00011502}, 
   {"Na", "I", "5", "2S -  9p  2Po", 2512.155, 2512.911, 0., 2, 4, 
    0.66667, 40499.99999999999, 0., 0.000076682}, 
   {"Na", "I", "5", "2S -  9p  2Po", 2512.134, 2512.89, 0., 2, 2, 
    0.33333, 40499.99999999999, 0., 0.000038341}, 
   {"Mg", "I", "", "1S -  3p  3Po", 4571.0968, 4572.3767, 
    0., 1, 3, 1., 218., 218., 2.0498*^-6}, 
   {"Mg", "I", "1", "1S -  3p  1Po", 2852.1262, 2852.9642, 0., 1, 3, 
    1., 5.*^8, 5.*^8, 1.8304}, {"Mg", "I", "", "1S -  4p  3Po", 
    2089.444, 2090.1082, 0., 1, 3, 1., 75., 0., 
    1.4736*^-7}, {"Mg", "I", "2", "1S -  4p  1Po", 2025.8242, 
    2026.4768, 0., 1, 3, 1., 6.25*^7, 7.25*^7, 0.11544}, 
   {"Mg", "I", "", "1S -  5p  1Po", Null, 1827.9351, 0., 1, 3, 1., 
    1.6399*^7, 0., 0.024496}, 
   {"Mg", "I", "", "1S -  6p  1Po", Null, 1747.7937, 0., 1, 3, 1., 
    6.8999*^6, 0., 0.0093426}, 
   {"Mg", "I", "", "1S -  7p  1Po", Null, 1707.0606, 0., 1, 3, 1., 3.16*^6, 
    0., 0.00414158}, {"Mg", "I", "", "1S -  8p  1Po", Null, 
    1683.4116, 0., 1, 3, 1., 1.95*^6, 0., 0.0024854}, 
   {"Mg", "I", "", "1S -  9p  1Po", Null, 1668.4288, 0., 1, 3, 1., 
    960299.9999999999, 0., 0.0012023}, 
   {"Mg", "I", "", "1S - 10p  1Po", Null, 1658.3115, 0., 1, 3, 1., 
    709070., 0., 0.00087799}, 
   {"Mg", "I", "", "1S - 11p  1Po", Null, 1651.1636, 0., 1, 3, 1., 536340., 
    0., 0.00065766}, {"Mg", "II", "1", "2S -  3p  2Po", 2797.916, 
    2798.7409, 0., 2, 6, 1., 2.6059*^8, 0., 
    0.91772}, {"Mg", "II", "1", "2S -  3p  2Po", 
    2802.705, 2803.531, 0., 2, 2, 0.33333, 
    2.5916*^8, 2.5916*^8, 0.30538}, 
   {"Mg", "II", "1", "2S -  3p  2Po", 2795.528, 2796.352, 0., 2, 4, 
    0.66667, 2.6117*^8, 2.6117*^8, 0.61234}, 
   {"Mg", "II", "", "2S -  4p  2Po", Null, 1240.0817, 0., 2, 6, 1., 
    579999.9999999999, 0., 0.00040115}, 
   {"Mg", "II", "", "2S -  4p  2Po", Null, 1240.3947, 0., 2, 2, 
    0.33333, 579560., 0., 0.00013368}, 
   {"Mg", "II", "", "2S -  4p  2Po", Null, 1239.9253, 0., 2, 4, 
    0.66667, 580220., 0., 0.00026747}, 
   {"Mg", "II", "", "2S -  5p  2Po", Null, 1026.0165, 0., 2, 6, 1., 
    2.7*^6, 0., 0.0012784}, 
   {"Mg", "II", "", "2S -  5p  2Po", Null, 1026.1134, 0., 2, 2, 0.33333, 
    2.6992*^6, 0., 0.00042607}, 
   {"Mg", "II", "", "2S -  5p  2Po", Null, 1025.9681, 0., 2, 4, 
    0.66667, 2.7004*^6, 0., 0.00085228}, 
   {"Mg", "II", "", "2S -  6p  2Po", Null, 946.7253, 0., 2, 6, 1., 
    2.6*^6, 0., 0.0010481}, 
   {"Mg", "II", "", "2S -  6p  2Po", Null, 946.7694, 0., 2, 2, 
    0.33333, 2.5996*^6, 0., 0.00034934}, 
   {"Mg", "II", "", "2S -  6p  2Po", Null, 946.7033, 0., 2, 4, 
    0.66667, 2.6002*^6, 0., 0.00069875}, 
   {"Al", "I", "1v", "2Po-  4s  2S ", 3955.665, 3956.7842, 
    74.707, 6, 2, 1., 1.4459*^8, 0., 0.11305}, 
   {"Al", "I", "1v", "2Po-  4s  2S ", 3961.52, 3962.641, 112.061, 
    4, 2, 0.66667, 9.5907*^7, 1.4459*^8, 
    0.11289}, {"Al", "I", "1v", "2Po-  4s  2S ", 3944.006, 3945.1224, 0., 2, 
    2, 0.33333, 4.8595*^7, 1.4459*^8, 
    0.11339}, {"Al", "I", "2v", "2Po-  3p2 4P ", 
    3458.2172, 3459.208, 112.061, 4, 2, Null, Null, Null, 
    Null}, {"Al", "I", "2v", "2Po-  3p2 4P ", 3452.658, 3453.647, 
    112.061, 4, 4, Null, Null, Null, Null}, 
   {"Al", "I", "2v", "2Po-  3p2 4P ", 3444.864, 3445.851, 0., 2, 2, 
    Null, Null, Null, Null}, {"Al", "I", "2v", "2Po-  3p2 4P ", 3443.6401, 
    3444.627, 112.031, 4, 6, Null, Null, Null, Null}, 
   {"Al", "I", "2v", "2Po-  3p2 4P ", 3439.3461, 3440.332, 0., 2, 
    4, Null, Null, Null, Null}, 
   {"Al", "I", "3v", "2Po-  3d  2D ", 3089.1911, 3090.0884, 
    74.707, 6, 10, 1., 7.14*^7, 0., 0.17035}, 
   {"Al", "I", "3v", "2Po-  3d  2D ", 3092.8383, 3093.7364, 
    112.061, 4, 4, 0.066679, 1.1859*^7, 
    7.1767*^7, 0.017016}, 
   {"Al", "I", "3v", "2Po-  3d  2D ", 3092.71, 3093.6079, 112.061, 
    4, 6, 0.6, 7.1157*^7, 7.1157*^7, 0.15314}, 
   {"Al", "I", "3v", "2Po-  3d  2D ", 3082.153, 3083.0479, 0., 2, 
    4, 0.33333, 5.9908*^7, 7.1767*^7, 0.17074}, 
   {"Al", "I", "1", "2Po-  5s  2S ", 2657.744, 2658.5347, 74.707, 
    6, 2, 1., 4.25*^7, 8.*^6, 0.015011}, 
   {"Al", "I", "1", "2Po-  5s  2S ", 2660.386, 2661.1774, 112.061, 
    4, 2, 0.66667, 2.8249*^7, 5.0599*^7, 0.014996}, 
   {"Al", "I", "1", "2Po-  5s  2S ", 2652.4753, 2653.265, 0., 2, 2, 
    0.33333, 1.4251*^7, 5.0599*^7, 0.015041}, 
   {"Al", "I", "2", "2Po-  4d  2D ", 2572.7392, 2573.5103, 
    74.707, 6, 10, 1., 2.3999*^7, 1.09*^7, 
    0.038061}, {"Al", "I", "2", "2Po-  4d  2D ", 
    2575.397, 2576.168, 112.061, 4, 4, 0.066679, 
    3.8217*^6, 3.3995*^7, 0.0038024}, 
   {"Al", "I", "2", "2Po-  4d  2D ", 2575.0944, 2575.8657, 112.061, 
    4, 6, 0.6, 2.2937*^7, 3.3837*^7, 0.034224}, 
   {"Al", "I", "2", "2Po-  4d  2D ", 2567.983, 2568.7523, 0., 2, 4, 
    0.33333, 1.9273*^7, 3.3995*^7, 0.038132}, 
   {"Al", "I", "3", "2Po-  6s  2S ", 2376.2822, 2377.008, 
    74.707, 6, 2, 1., 1.43*^7, 0., 0.0040377}, 
   {"Al", "I", "3", "2Po-  6s  2S ", 2378.395, 2379.1204, 112.061, 
    4, 2, 0.66667, 9.5089*^6, 0., 0.0040341}\
, {"Al", "I", "3", 
      "2Po-  6s  2S ", 2372.07, 2372.7944, 0., 2, 2, 
    0.33333, 4.7921*^6, 0., 0.0040448}, 
   {"Al", "I", "4", "2Po-  5d  2D ", 2371.1103, 2371.8349, 
    74.707, 6, 10, 1., 7.5*^7, 759999.9999999999, 0.10542}, 
   {"Al", "I", "4", "2Po-  5d  2D ", 2373.351, 2374.0754, 112.061, 4, 4, 
    0.066679, 1.2465*^7, 7.6047*^7, 
    0.010533}, {"Al", "I", "4", "2Po-  5d  2D ", 
    2373.122, 2373.8467, 112.061, 4, 6, 0.6, 7.4809*^7, 7.5569*^7, 
    0.0948}, {"Al", "I", "4", "2Po-  5d  2D ", 2367.053, 
    2367.7762, 0., 2, 4, 0.33333, 6.2821*^7, 7.6047*^7, 
    0.1056}, {"Al", "I", "5", "2Po-  6d  2D ", 2267.2231, 2267.9246, 
    74.707, 6, 10, 1., 6.9399*^7, 2.1*^6, 
    0.089063}, {"Al", "I", "5", "2Po-  6d  2D ", 2269.2214, 
    2269.9231, 112.061, 4, 4, 0.066679, 1.152*^7, 
    7.1658*^7, 0.0088989}, 
   {"Al", "I", "5", "2Po-  6d  2D ", 2269.0954, 2269.7971, 112.061, 
    4, 6, 0.6, 6.91298*^7, 7.1229*^7, 0.08009}, 
   {"Al", "I", "5", "2Po-  6d  2D ", 2263.4633, 2264.1637, 0., 2, 4, 
    0.33333, 5.8038*^7, 7.1658*^7, 0.08921}\
, {"Al", "I", "6", 
      "2Po-  7s  2S ", 2261.8242, 2262.5245, 74.707, 
    6, 2, 1., 1.1399*^7, 0., 0.0028907}, 
   {"Al", "I", "6", "2Po-  7s  2S ", 2263.738, 2264.4383, 112.061, 
    4, 2, 0.66667, 7.5143*^6, 0., 0.0028883}, 
   {"Al", "I", "6", "2Po-  7s  2S ", 2258.008, 2258.7068, 
    0., 2, 2, 0.33333, 3.7858*^6, 0., 0.0028955}, 
   {"Al", "I", "7", "2Po-  7d  2D ", 2208.264, 2208.953, 74.707, 
    6, 10, 1., 4.4699*^7, 2.3999*^6, 
    0.054377}, {"Al", "I", "7", "2Po-  7d  2D ", 
    2210.1301, 2210.8192, 112.061, 4, 4, 
    0.066679, 7.4149*^6, 4.7063*^7, 0.00543348}, 
   {"Al", "I", "7", "2Po-  7d  2D ", 2210.06, 2210.7491, 112.061, 
    4, 6, 0.6, 4.4491*^7, 4.6791*^7, 0.048899}, 
   {"Al", "I", "7", "2Po-  7d  2D ", 2204.668, 2205.3555, 0., 2, 4, 0.33333, 
    3.7348*^7, 4.7063*^7, 0.054465}, 
   {"Al", "I", "8", "2Po-  8s  2S ", 2202.8034, 2203.4912, 
    74.707, 6, 2, 1., 5.2*^6, 0., 0.0012617}, 
   {"Al", "I", "8", "2Po-  8s  2S ", 2204.619, 2205.3064, 112.061, 4, 2, 
    0.66667, 3.4581*^6, 0., 0.0012607}, 
   {"Al", "I", "8", "2Po-  8s  2S ", 2199.1831, 2199.8699, 0., 2, 2, 
    0.33333, 1.7419*^6, 0., 0.0012638}, 
   {"Al", "I", "9", "2Po-  8d  2D ", 2172.3222, 2173.0036, 
    74.707, 6, 10, 1., 3.11*^7, 3.*^6, 0.036693}, 
   {"Al", "I", "9", "2Po-  8d  2D ", 2174.113, 2174.7942, 112.061, 
    4, 4, 0.066679, 5.1708*^6, 3.4213*^7, 0.0036665}, 
   {"Al", "I", "9", "2Po-  8d  2D ", 2174.071, 2174.752, 112.061, 4, 6, 0.6, 
    3.1025*^7, 3.4025*^7, 0.032997}, 
   {"Al", "I", "9", "2Po-  8d  2D ", 2168.8264, 2169.5069, 
    0., 2, 4, 0.33333, 2.6042*^7, 3.4213*^7, 0.036752}, 
   {"Al", "I", "", "2Po-  9s  2S ", 2168.0844, 2168.7651, 
    74.707, 6, 2, 1., 3.7913*^6, 0., 0.00089115}, 
   {"Al", "I", "", "2Po-  9s  2S ", 2169.843, 2170.5235, 
    112.061, 4, 2, 0.66719, 2.5234*^6, 0., 0.00089113}, 
   {"Al", "I", "", "2Po-  9s  2S ", 2164.5773, 2165.2569, 0., 2, 2, 
    0.33281, 1.2679*^6, 0., 0.00089119}, 
   {"Al", "I", "", "2Po-  9d  2D ", 2148.9833, 2149.6601, 
    74.707, 6, 10, 1., 1.53*^7, 6.6999*^6, 0.017666}, 
   {"Al", "I", "", "2Po-  9d  2D ", 2150.7282, 2151.405, 112.061, 4, 4, 
    0.066679, 2.5439*^6, 2.1955*^7, 0.0017652}, 
   {"Al", "I", "", "2Po-  9d  2D ", 2150.6992, 2151.376, 112.061, 
    4, 6, 0.6, 1.5263*^7, 2.1863*^7, 0.015887}, 
   {"Al", "I", "", "2Po-  9d  2D ", 2145.5541, 2146.23, 0., 2, 4, 0.33333, 
    1.2811*^7, 2.1955*^7, 0.017694}, 
   {"Al", "I", "", "2Po- 10d  2D ", 2133.042, 2133.7154, 
    74.707, 6, 10, 1., 1.82*^7, 0., 
    0.020704}, {"Al", "I", "", "2Po- 10d  2D ", 2134.76, 
    2135.433, 112.061, 4, 4, 0.066679, 3.0262*^6, 0., 
    0.0020688}, {"Al", "I", "", "2Po- 10d  2D ", 
    2134.7333, 2135.407, 112.061, 4, 6, 0.6, 1.8157*^7, 0., 
    0.018619}, {"Al", "I", "", "2Po- 10d  2D ", 
    2129.663, 2130.335, 0., 2, 4, 0.33333, 1.5239*^7, 
    0., 0.020736}, {"Al", "I", "", "2Po- 11d  2D ", 2121.678, 
    2122.3491, 74.707, 6, 10, 1., 1.2399*^7, 0., 
    0.013843}, {"Al", "I", "", "2Po- 11d  2D ", 2123.362, 2124.033, 
    112.061, 4, 4, 0.066679, 2.0452*^6, 0., 0.0013833}, 
   {"Al", "I", "", "2Po- 11d  2D ", 2123.362, 2124.033, 112.061, 4, 
    6, 0.6, 1.2271*^7, 0., 0.012449}, 
   {"Al", "I", "", "2Po- 11d  2D ", 2118.32, 2118.99, 0., 2, 4, 
    0.33333, 1.0299*^7, 0., 0.013865}, 
   {"Al", "I", "Auto", "2Po-  3p2 2S ", Null, 1935.0603, 
    74.707, 6, 2, 1., 4.1799*^8, 6.49*^12, 
    0.078039}, {"Al", "I", "Auto", "2Po-  3p2 2S ", Null, 
    1936.46, 112.061, 4, 2, 0.66667, 2.774*^8, 
    6.4904*^12, 0.077974}, 
   {"Al", "I", "Auto", "2Po-  3p2 2S ", Null, 1932.27, 0., 2, 2, 0.33333, 
    1.3969*^8, 6.4904*^12, 0.0781428}, 
   {"Al", "I", "Auto", "2Po-  3p2 2P ", Null, 1766.1315, 
    74.707, 6, 6, 1., 1.85*^9, 0., 0.86512}, 
   {"Al", "I", "Auto", "2Po-  3p2 2P ", Null, 1769.133, 112.061, 4, 2, 
    0.11111, 6.1353*^8, 1.06*^10, 0.14394}, 
   {"Al", "I", "Auto", "2Po-  3p2 2P ", Null, 1766.381, 112.061, 4, 4, 
    0.55556, 1.541*^9, 4.1899*^10, 0.72083}\
, {"Al", "I", "Auto", 
      "2Po-  3p2 2P ", Null, 1765.632, 0., 2, 2, 
    0.22222, 1.2344*^9, 1.06*^10, 0.5769}, 
   {"Al", "I", "Auto", "2Po-  3p2 2P ", Null, 1762.892, 0., 2, 4, 
    0.11111, 3.1003*^8, 4.1899*^10, 0.28899}\
, {"Al", "II", "", 
      "1S -  3p  3Po", 2669.1542, 2669.948, 0., 1, 3, 
    1., 3330.99, 3330.99, 0.000010676}, 
   {"Al", "II", "2", "1S -  3p  1Po", Null, 1670.7874, 0., 1, 3, 1., 
    1.4699*^9, 1.4699*^9, 1.833}, 
   {"Al", "II", "", "1S -  4p  1Po", Null, 935.2738, 0., 1, 3, 1., 
    6.6399*^7, 0., 0.026084}, 
   {"Al", "III", "1", "2S -  3p  2Po", Null, 1857.3996, 0., 2, 6, 1., 
    5.4089*^8, 0., 0.83912}, 
   {"Al", "III", "1", "2S -  3p  2Po", Null, 1862.7895, 0., 2, 2, 0.33333, 
    5.3611*^8, 5.3611*^8, 0.2789}, 
   {"Al", "III", "1", "2S -  3p  2Po", Null, 1854.7164, 0., 2, 4, 
    0.66667, 5.4315*^8, 5.4315*^8, 
    0.56023}, {"Si", "I", "0.01", "3P -  3p3 5So", 3020.004, 
    3020.8837, 223.157, 5, 5, Null, 2459.8, 0., 
    3.3653*^-6}, {"Si", "I", "0.01", "3P -  3p3 5So", 3006.739, 
    3007.6149, 77.115, 3, 5, Null, 842.81, 0., 
    1.9049*^-6}, {"Si", "I", "1", "3P -  4s  3Po", 2517.484, 
    2518.2422, 149.681, 9, 9, 1., 1.6101*^8, 0., 
    0.15308}, {"Si", "I", "1", "3P -  4s  3Po", 
    2528.508, 2529.2682, 223.157, 5, 3, 0.14546, 6.9348*^7, 
    1.6845*^8, 0.039905}, 
   {"Si", "I", "1", "3P -  4s  3Po", 2524.108, 2524.8669, 77.115, 
    3, 1, 0.11015, 1.5837*^8, 1.5837*^8, 
    0.050452}, {"Si", "I", "1", "3P -  4s  3Po", 2519.202, 
    2519.96, 77.115, 3, 3, 0.08733, 4.2095*^7, 
    1.6845*^8, 0.040075}, 
   {"Si", "I", "1", "3P -  4s  3Po", 2516.1123, 2516.8696, 223.157, 
    5, 5, 0.40796, 1.1843*^8, 1.5703*^8, 0.11247}, 
   {"Si", "I", "1", "3P -  4s  3Po", 2514.3154, 2515.0725, 0., 1, 3, 
    0.11757, 5.7006*^7, 1.6845*^8, 
    0.16218}, {"Si", "I", "1", "3P -  4s  3Po", 
    2506.897, 2507.6522, 77.115, 3, 5, 0.13153, 
    3.8605*^7, 1.5703*^8, 0.060657}, 
   {"Si", "I", "2", "3P -  4s  1Po", 2452.118, 2452.8605, 
    223.157, 5, 3, Null, 672480., 2.6532*^8, 0.00036394}, 
   {"Si", "I", "2", "3P -  4s  1Po", 2443.365, 2444.1052, 
    77.115, 3, 3, Null, 725740., 2.6532*^8, 0.00064995}, 
   {"Si", "I", "2", "3P -  4s  1Po", 2438.768, 2439.5073, 0., 1, 3, Null, 
    917099.9999999999, 2.6532*^8, 0.0024547}, 
   {"Si", "I", "3", "3P -  3p3 3Do", 2213.9711, 2214.6615, 149.681, 9, 15, 
    1., 3.1282*^7, 0., 0.038337}, 
   {"Si", "I", "3", "3P -  3p3 3Do", 2218.916, 2219.6065, 223.157, 
    5, 3, 0.00553, 859940., 3.6058*^7, 
    0.00038109}, {"Si", "I", "3", "3P -  3p3 3Do", 
    2218.057, 2218.7476, 223.157, 5, 5, 
    0.069659, 6.5007*^6, 2.7191*^7, 0.0047977}, 
   {"Si", "I", "3", "3P -  3p3 3Do", 2216.669, 2217.3593, 223.157, 5, 7, 
    0.48158, 3.2164*^7, 3.2164*^7, 0.033192}, 
   {"Si", "I", "3", "3P -  3p3 3Do", 2211.746, 2212.4348, 77.115, 
    3, 3, 0.091568, 1.4364*^7, 3.6058*^7, 
    0.010541}, {"Si", "I", "3", "3P -  3p3 3Do", 2210.8923, 
    2211.5814, 77.115, 3, 5, 0.21955, 2.0699*^7, 
    2.7191*^7, 0.025286}, 
   {"Si", "I", "3", "3P -  3p3 3Do", 2207.978, 2208.6665, 0., 1, 3, 
    0.13212, 2.0833*^7, 3.6058*^7, 0.045709}\
, {"Si", "I", "4", 
      "3P -  3d  1Do", 2121.1921, 2121.863, 223.157, 
    5, 5, Null, 141820., 5.9167*^7, 
    0.000095726}, {"Si", "I", "4", "3P -  3d  1Do", 
    2114.6384, 2115.308, 77.115, 3, 5, Null, 
    25376., 5.9167*^7, 0.000028371}, 
   {"Si", "I", "6", "3P -  3d  3Fo", 2014.3542, 2015.0048, 223.157, 
    5, 5, Null, 1003.7, 0., 6.1098*^-7}, 
   {"Si", "I", "6", "3P -  3d  3Fo", 2010.993, 2011.6427, 223.157, 5, 7, 
    Null, 7276.9, 0., 6.1806*^-6}, 
   {"Si", "I", "6", "3P -  3d  3Fo", 2008.4431, 2009.0926, 
    77.115, 3, 5, Null, 5850.2, 0., 5.9004*^-6}, 
   {"Si", "I", "7", "3P -  3d  3Po", Null, 1984.7672, 149.681, 9, 9, 
    1., 6.5965*^7, 0., 0.038957}, 
   {"Si", "I", "7", "3P -  3d  3Po", Null, 1988.9937, 223.157, 5, 5, 
    0.42276, 4.9878*^7, 6.6499*^7, 0.029582}, 
   {"Si", "I", "7", "3P -  3d  3Po", Null, 1986.364, 223.157, 5, 3, 
    0.14639, 2.8999*^7, 6.7365*^7, 0.010257}\
, {"Si", "I", "7", 
      "3P -  3d  3Po", Null, 1983.2328, 77.115, 3, 
    5, 0.13958, 1.6612*^7, 6.6499*^7, 0.016326}, 
   {"Si", "I", "7", "3P -  3d  3Po", Null, 1980.6184, 77.115, 3, 
    3, 0.08208, 1.6346*^7, 6.7365*^7, 0.0096134}, 
   {"Si", "I", "7", "3P -  3d  3Po", Null, 1979.2055, 
    77.115, 3, 1, 0.09862, 5.9042*^7, 
    5.9042*^7, 0.011558}, 
   {"Si", "I", "7", "3P -  3d  3Po", Null, 1977.5979, 0., 1, 3, 
    0.11056, 2.2118*^7, 6.7365*^7, 0.038905}, 
   {"Si", "I", "8", "3P -  3d  1Fo", Null, 1881.8541, 223.157, 5, 7, 
    Null, 5.6218*^6, 3.3562*^8, 0.0041786}, 
   {"Si", "I", "9", "3P -  3d  1Po", Null, 1880.9568, 223.157, 5, 3, 
    Null, 300790., 0., 0.000095726}, 
   {"Si", "I", "9", "3P -  3d  1Po", Null, 1875.813, 77.115, 3, 
    3, Null, 2.242*^6, 0., 0.0011827}, 
   {"Si", "I", "9", "3P -  3d  1Po", Null, 1873.1035, 0., 1, 3, 
    Null, 1.7057*^6, 0., 0.0026915}, 
   {"Si", "I", "10", "3P -  3d  3Do", Null, 1849.2512, 149.681, 9, 
    15, 1., 2.3514*^8, 0., 0.20092}, 
   {"Si", "I", "10", "3P -  3d  3Do", Null, 1853.1522, 223.157, 5, 3, 
    0.00801, 9.3584*^6, 0., 0.0028909}, 
   {"Si", "I", "10", "3P -  3d  3Do", Null, 1852.4716, 223.157, 5, 5, 
    0.087899, 6.1613*^7, 0., 0.031698}, 
   {"Si", "I", "10", "3P -  3d  3Do", Null, 1850.672, 223.157, 5, 7, 
    0.43962, 2.21*^8, 0., 0.15887}, 
   {"Si", "I", "10", "3P -  3d  3Do", Null, 1848.1505, 
    77.115, 3, 3, 0.0856, 1.0082*^8, 0., 0.051627}, 
   {"Si", "I", "10", "3P -  3d  3Do", Null, 1847.4735, 77.115, 3, 
    5, 0.25254, 1.7866*^8, 0., 0.15236}, 
   {"Si", "I", "10", "3P -  3d  3Do", Null, 1845.5202, 0., 1, 3, 
    0.12643, 1.4955*^8, 0., 0.22909}, 
   {"Si", "I", "11", "3P -  5s  3Po", Null, 1842.4484, 149.681, 9, 
    9, Null, Null, Null, Null}, 
   {"Si", "I", "11", "3P -  5s  3Po", Null, 1848.748, 223.157, 5, 
    3, Null, 3.9198*^7, 0., 0.012051}, 
   {"Si", "I", "11", "3P -  5s  3Po", Null, 1846.1116, 
    77.115, 3, 1, Null, Null, Null, Null}, 
   {"Si", "I", "11", "3P -  5s  3Po", Null, 1843.7699, 77.115, 3, 
    3, Null, 3.3544*^7, 0., 0.017095}, 
   {"Si", "I", "11", "3P -  5s  3Po", Null, 1841.4489, 223.157, 5, 
    5, Null, 3.6716*^7, 0., 0.018665}, 
   {"Si", "I", "11", "3P -  5s  3Po", Null, 1841.1521, 0., 1, 3, Null, 
    1.5026*^7, 0., 0.022909}, 
   {"Si", "I", "11", "3P -  5s  3Po", Null, 1836.51, 77.115, 3, 
    5, Null, 1.3097*^7, 0., 0.011038}, 
   {"Si", "I", "12", "3P -  5s  1Po", Null, 1827.4403, 149.681, 9, 
    3, 1., 5.7679*^6, 0., 0.00096259}, 
   {"Si", "I", "12", "3P -  5s  1Po", Null, 1829.8974, 223.157, 5, 3, 
    0.21033, 1.2083*^6, 0., 0.00036394}, 
   {"Si", "I", "12", "3P -  5s  1Po", Null, 1825.0201, 
    77.115, 3, 3, 0.06335, 366850., 0., 0.00018318}, 
   {"Si", "I", "12", "3P -  5s  1Po", Null, 1822.4553, 0., 1, 3, 
    0.72632, 4.2238*^6, 0., 0.0063096}, 
   {"Si", "I", "12", "3P -  5s  1Po", Null, 1822.4553, 0., 1, 3, Null, 
    4.2238*^6, 0., 0.0063096}, 
   {"Si", "I", "13", "3P -  4d  1Do", Null, 1776.8242, 223.157, 5, 5, Null, 
    5.8329*^6, 0., 0.0027608}, {"Si", "I", "13", "3P -  4d  1Do", Null, 
    1772.2254, 77.115, 3, 5, Null, 238860., 0., 
    0.00018745}, {"Si", "I", "14", "3P -  4d  3Po", Null, 1768.3761, 
    149.681, 9, 9, 1., 1.7741*^7, 0., 0.0083173}, 
   {"Si", "I", "14", "3P -  4d  3Po", Null, 1770.9225, 223.157, 5, 
    5, 0.35188, 1.1189*^7, 0., 0.0052605}, 
   {"Si", "I", "14", "3P -  4d  3Po", Null, 1770.6294, 223.157, 5, 
    3, 0.212, 1.124*^7, 0., 0.0031698}, 
   {"Si", "I", "14", "3P -  4d  3Po", Null, 1766.3542, 
    77.115, 3, 5, 0.048459, 1.5524*^6, 0., 
    0.0012103}, {"Si", "I", "14", "3P -  4d  3Po", Null, 
    1766.0626, 77.115, 3, 3, 0.17588, 9.3974*^6, 0., 
    0.0043942}, {"Si", "I", "14", "3P -  4d  3Po", Null, 1765.0299, 
    77.115, 3, 1, 0.10359, 1.662*^7, 0., 
    0.0025875}, {"Si", "I", "14", "3P -  4d  3Po", Null, 
    1763.6607, 0., 1, 3, 0.10839, 5.8102*^6, 0., 
    0.0081283}, {"Si", "I", "14", "3P -  4d  3Po", Null, 
    1770.9225, 223.157, 5, 5, Null, 1.1189*^7, 0., 0.0052605}, 
   {"Si", "I", "14", "3P -  4d  3Po", Null, 1763.6607, 0., 1, 3, 
    Null, 5.8102*^6, 0., 0.0081283}, 
   {"Si", "I", "15", "3P -  4d  3Fo", Null, 1749.8076, 223.157, 5, 5, Null, 
    Null, Null, Null}, {"Si", "I", "15", "3P -  4d  3Fo", Null, 
    1747.414, 223.157, 5, 7, Null, 1.5285*^6, 0., 0.00097956}, 
   {"Si", "I", "15", "3P -  4d  3Fo", Null, 1745.3475, 77.115, 3, 
    5, Null, Null, Null, Null}, 
   {"Si", "I", "16", "3P -  4d  1Po", Null, 1707.1147, 223.157, 5, 3, Null, 
    Null, Null, Null}, {"Si", "I", "16", "3P -  4d  1Po", Null, 1702.8693, 
    77.115, 3, 3, Null, 1.8822*^7, 0., 
    0.0081824}, {"Si", "I", "16", "3P -  4d  1Po", Null, 
    1700.636, 0., 1, 3, Null, 3.0605*^7, 0., 0.039811}, 
   {"Si", "I", "17", "3P -  4d  1Fo", Null, 1704.4416, 223.157, 5, 7, Null, 
    2.6054*^7, 0., 0.015887}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1697.0026, 149.681, 9, 15, 1., 
    1.3716*^8, 0., 0.098694}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1700.4191, 223.157, 5, 
    5, 0.07627, 3.1193*^7, 0., 0.013522}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1699.7162, 223.157, 5, 
    3, 0.00264, 1.8041*^6, 0., 0.00046885}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1697.9407, 223.157, 5, 7, 
    0.46958, 1.3778*^8, 0., 0.083374}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1696.2068, 
    77.115, 3, 5, 0.29598, 1.2196*^8, 0., 0.087676}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1695.5074, 
    77.115, 3, 3, 0.05769, 3.9666*^7, 0., 0.017095}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1693.2935, 0., 1, 3, 
    0.097848, 6.7539*^7, 0., 0.087096}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1700.4191, 223.157, 5, 
    5, Null, 3.1193*^7, 0., 0.013522}, 
   {"Si", "I", "18", "3P -  4d  3Do", Null, 1697.9407, 223.157, 5, 7, Null, 
    1.3778*^8, 0., 0.083374}, {"Si", "I", "18", "3P -  4d  3Do", Null, 
    1696.2068, 77.115, 3, 5, Null, 1.2196*^8, 0., 
    0.087676}, {"Si", "I", "18", "3P -  4d  3Do", Null, 
    1695.5074, 77.115, 3, 3, Null, 3.9666*^7, 0., 
    0.017095}, {"Si", "I", "18", "3P -  4d  3Do", Null, 
    1693.2935, 0., 1, 3, Null, 6.7539*^7, 0., 0.087096}, 
   {"Si", "I", "", "3P -(1/2,1/2)", Null, 1693.4681, 223.157, 5, 3, Null, 
    1.2008*^7, 0., 0.0030976}, 
   {"Si", "I", "", "3P -(1/2,1/2)", Null, 1690.7887, 
    77.115, 3, 1, Null, 2.8705*^7, 0., 0.0041009}, 
   {"Si", "I", "", "3P -(1/2,1/2)", Null, 1689.2902, 
    77.115, 3, 3, Null, 1.1524*^7, 0., 0.0049304}, 
   {"Si", "I", "", "3P -(1/2,1/2)", Null, 1687.0924, 0., 1, 3, Null, 
    3.6538*^6, 0., 0.0046774}, 
   {"Si", "I", "", "3P -(3/2,1/2)", Null, 1686.8185, 223.157, 5, 5, Null, 
    1.5347*^7, 0., 0.0065468}, 
   {"Si", "I", "", "3P -(3/2,1/2)", Null, 1683.1189, 223.157, 5, 3, Null, 
    1.0466*^6, 0., 0.00026679}, 
   {"Si", "I", "", "3P -(3/2,1/2)", Null, 1682.6733, 
    77.115, 3, 5, Null, 5.1543*^6, 0., 0.0036465}, 
   {"Si", "I", "", "3P -(3/2,1/2)", Null, 1678.9918, 
    77.115, 3, 3, Null, 753230., 0., 0.00031833}, 
   {"Si", "I", "", "3P -(3/2,1/2)", Null, 1676.8207, 0., 1, 3, Null, 
    941999.9999999999, 0., 0.0011912}, 
   {"Si", "I", "23", "3P -   a  3Po", Null, 1671.8884, 149.681, 9, 
    9, Null, Null, Null, Null}, 
   {"Si", "I", "23", "3P -   a  3Po", Null, 1675.2052, 223.157, 5, 
    5, Null, 6.4126*^7, 0., 0.026979}, 
   {"Si", "I", "23", "3P -   a  3Po", Null, 1672.596, 223.157, 5, 
    3, Null, 3.092*^7, 0., 0.0077809}, 
   {"Si", "I", "23", "3P -   a  3Po", Null, 1671.1168, 77.115, 3, 
    5, Null, 8.3016*^6, 0., 0.0057927}, 
   {"Si", "I", "23", "3P -   a  3Po", Null, 1668.5203, 
    77.115, 3, 3, Null, 2.5844*^7, 0., 0.010786}, 
   {"Si", "I", "23", "3P -   a  3Po", Null, 1667.6287, 
    77.115, 3, 1, Null, Null, Null, Null}, 
   {"Si", "I", "23", "3P -   a  3Po", Null, 1666.3762, 0., 1, 3, 
    Null, 1.7518*^7, 0., 0.021878}, 
   {"Si", "I", "24", "3P -  5d  1Do", Null, 1664.511, 223.157, 5, 
    5, Null, 7.6313*^6, 0., 0.0031698}, 
   {"Si", "I", "24", "3P -  5d  1Do", Null, 1660.4746, 
    77.115, 3, 5, Null, Null, Null, Null}, 
   {"Si", "I", "25", "3P -  5d  3Fo", Null, 1655.0185, 223.157, 5, 
    5, Null, Null, Null, Null}, 
   {"Si", "I", "25", "3P -  5d  3Fo", Null, 1653.3761, 223.157, 5, 
    7, Null, 5.0385*^6, 0., 0.0028909}, 
   {"Si", "I", "25", "3P -  5d  3Fo", Null, 1651.028, 77.115, 3, 
    5, Null, 3.3858*^6, 0., 0.0023061}, 
   {"Si", "I", "26", "3P -  5d  1Po", Null, 1633.2249, 
    77.115, 3, 3, Null, 6.19348*^7, 0., 0.024767}, 
   {"Si", "I", "26", "3P -  5d  1Po", Null, 1631.1705, 0., 1, 3, Null, 
    1.7149*^8, 0., 0.20512}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1629.4552, 149.681, 9, 15, 1., 
    2.5907*^8, 0., 0.17187}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1633.3276, 223.157, 5, 5, 
    0.040699, 3.1335*^7, 0., 0.012532}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1631.6245, 223.157, 5, 
    3, 0.00392, 5.0557*^6, 0., 0.0012107}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1629.9478, 223.157, 5, 7, 
    0.48381, 2.6835*^8, 0., 0.14963}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1629.4409, 77.115, 3, 
    5, 0.38774, 3.0137*^8, 0., 0.19993}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1627.7458, 77.115, 3, 
    3, 0.03991, 5.1862*^7, 0., 0.020601}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1625.7051, 0., 1, 3, 
    0.04401, 5.7404*^7, 0., 0.068234}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1633.3276, 223.157, 5, 5, Null, 
    3.1335*^7, 0., 0.012532}, {"Si", "I", "27", "3P -  5d  3Do", Null, 
    1629.9478, 223.157, 5, 7, Null, 2.6835*^8, 0., 
    0.14963}, {"Si", "I", "27", "3P -  5d  3Do", Null, 1629.4409, 
    77.115, 3, 5, Null, 3.0137*^8, 0., 0.19993}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1627.7458, 77.115, 3, 
    3, Null, 5.1862*^7, 0., 0.020601}, 
   {"Si", "I", "27", "3P -  5d  3Do", Null, 1625.7051, 0., 1, 3, 
    Null, 5.7404*^7, 0., 0.068234}, 
   {"Si", "I", "28", "3P -  5d  1Fo", Null, 1633.985, 223.157, 5, 
    7, Null, 6.4498*^7, 0., 0.036143}, 
   {"Si", "I", "30", "3P -  5d  3Po", Null, 1619.7796, 149.681, 9, 9, 1., 
    2.2542*^8, 0., 0.088666}, 
   {"Si", "I", "30", "3P -  5d  3Po", Null, 1622.882, 223.157, 5, 5, 
    0.51503, 2.0778*^8, 0., 0.082041}, 
   {"Si", "I", "30", "3P -  5d  3Po", Null, 1620.405, 223.157, 5, 3, 
    0.15107, 1.0204*^8, 0., 0.024101}, 
   {"Si", "I", "30", "3P -  5d  3Po", Null, 1619.044, 77.115, 3, 
    5, 0.019679, 7.992*^6, 0., 0.0052345}, 
   {"Si", "I", "30", "3P -  5d  3Po", Null, 1616.579, 
    77.115, 3, 3, 0.13588, 9.2433*^7, 0., 
    0.036214}, {"Si", "I", "30", "3P -  5d  3Po", Null, 
    1615.949, 77.115, 3, 1, 0.11939, 
    2.4394*^8, 0., 0.031833}, {"Si", "I", "30", "3P -  5d  3Po", Null, 
    1614.566, 0., 1, 3, 0.05897, 4.0263*^7, 0., 0.047206}\
, {"Si", "I", 
      "30", "3P -  5d  3Po", Null, 1622.882, 223.157, 5, 5, Null, 
    2.0778*^8, 0., 0.082041}, {"Si", "I", "30", "3P -  5d  3Po", Null, 
    1620.405, 223.157, 5, 3, Null, 1.0204*^8, 0., 
    0.024101}, {"Si", "I", "30", "3P -  5d  3Po", Null, 
    1616.579, 77.115, 3, 3, Null, 9.2433*^7, 
    0., 0.036214}, {"Si", "I", "30", "3P -  5d  3Po", Null, 
    1615.949, 77.115, 3, 1, Null, 2.4394*^8, 0., 
    0.031833}, {"Si", "I", "30", "3P -  5d  3Po", Null, 1614.566, 
    0., 1, 3, Null, 4.0263*^7, 0., 0.047206}, 
   {"Si", "I", "35.02", "3P -  6d  3Do", Null, 1594.95, 77.115, 
    3, 5, Null, 2.5685*^8, 0., 0.16326}, 
   {"Si", "I", "35.02", "3P -  6d  3Do", Null, 1591.123, 77.115, 
    3, 3, Null, 2.7581*^7, 0., 0.010468}, 
   {"Si", "I", "35.02", "3P -  6d  3Do", Null, 1589.173, 0., 1, 3, Null, 
    1.4746*^7, 0., 0.016749}, {"Si", "I", "35.03", "3P -  6d  3Po", Null, 
    1589.6272, 149.681, 9, 9, 1., 2.138*^8, 0., 0.080995}, 
   {"Si", "I", "35.03", "3P -  6d  3Po", Null, 1592.426, 223.157, 
    5, 5, 0.51649, 1.9772*^8, 0., 0.075167}, 
   {"Si", "I", "35.03", "3P -  6d  3Po", Null, 1590.477, 223.157, 5, 3, 
    0.15507, 9.9304*^7, 0., 0.022596}, 
   {"Si", "I", "35.03", "3P -  6d  3Po", Null, 1588.731, 
    77.115, 3, 5, 0.00045, 172610., 0., 0.00010886}, 
   {"Si", "I", "35.03", "3P -  6d  3Po", Null, 1586.792, 
    77.115, 3, 3, 0.16549, 1.0666*^8, 0., 
    0.040269}, {"Si", "I", "35.03", "3P -  6d  3Po", Null, 
    1586.137, 77.115, 3, 1, 0.12687, 2.4573*^8, 0., 
    0.030894}, {"Si", "I", "35.03", "3P -  6d  3Po", Null, 
    1584.852, 0., 1, 3, 0.035739, 2.3123*^7, 0., 
    0.026122}, {"Si", "I", "35.03", "3P -  6d  3Po", Null, 
    1592.426, 223.157, 5, 5, Null, 1.9772*^8, 0., 
    0.075167}, {"Si", "I", "35.03", "3P -  6d  3Po", Null, 
    1590.477, 223.157, 5, 3, Null, 9.9304*^7, 0., 0.022596}, 
   {"Si", "I", "35.03", "3P -  6d  3Po", Null, 1586.792, 
    77.115, 3, 3, Null, 1.0666*^8, 0., 
    0.040269}, {"Si", "I", "35.03", "3P -  6d  3Po", Null, 
    1586.137, 77.115, 3, 1, Null, 2.4573*^8, 0., 
    0.030894}, {"Si", "I", "35.03", "3P -  6d  3Po", Null, 
    1584.852, 0., 1, 3, Null, 2.3123*^7, 0., 0.026122}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1573.6393, 149.681, 9, 15, 1., 
    1.3485*^8, 0., 0.08344}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1578.477, 223.157, 5, 
    5, 0.00014999, 57769., 0., 0.000021579}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1574.847, 77.115, 3, 
    5, 0.48053, 1.9395*^8, 0., 0.12019}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1574.128, 223.157, 5, 
    3, 0.00555, 3.7406*^6, 0., 0.00083374}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1573.884, 223.157, 5, 7, 
    0.48134, 1.3903*^8, 0., 0.072282}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1570.518, 77.115, 3, 
    3, 0.02288, 1.5522*^7, 0., 0.0057396}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1568.618, 0., 1, 3, 
    0.00955, 6.5011*^6, 0., 0.0071945}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1574.847, 77.115, 3, 
    5, Null, 1.9395*^8, 0., 0.12019}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1573.884, 223.157, 5, 7, Null, 
    1.3903*^8, 0., 0.072282}, {"Si", "I", "40", "3P -  7d  3Do", Null, 
    1570.518, 77.115, 3, 3, Null, 1.5522*^7, 0., 0.0057396}, 
   {"Si", "I", "40", "3P -  7d  3Do", Null, 1568.618, 0., 1, 3, 
    Null, 6.5011*^6, 0., 0.0071945}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1567.465, 223.157, 5, 3, 
    Null, 1.1578*^7, 0., 0.0025588}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1563.885, 
    77.115, 3, 3, Null, 3.7724*^7, 0., 0.013832}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1562.002, 0., 1, 3, Null, 
    3.4259*^8, 0., 0.37584}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1563.885, 
    77.115, 3, 3, Null, 3.7724*^7, 0., 0.013832}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1562.002, 0., 1, 3, Null, 
    3.4259*^8, 0., 0.37584}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1567.465, 223.157, 5, 3, 
    Null, 1.1578*^7, 0., 0.0025588}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1563.885, 
    77.115, 3, 3, Null, 3.7724*^7, 0., 0.013832}, 
   {"Si", "I", "", "3P -  8d  1Po", Null, 1562.002, 0., 1, 3, Null, 
    3.4259*^8, 0., 0.37584}, 
   {"Si", "I", "", "3P-(3/2,5/2)o", Null, 1560.932, 223.157, 5, 3, Null, 
    8.9589*^6, 0., 0.0019635}, 
   {"Si", "I", "", "3P-(3/2,5/2)o", Null, 1560.742, 223.157, 5, 7, 
    Null, 2.9267*^8, 0., 0.14963}, 
   {"Si", "I", "", "3P-(3/2,5/2)o", Null, 1557.382, 77.115, 3, 3, 
    Null, 5.0145*^7, 0., 0.018234}, 
   {"Si", "I", "", "3P-(3/2,5/2)o", Null, 1555.514, 0., 1, 3, Null, 
    3.1065*^7, 0., 0.033806}, 
   {"Si", "I", "", "3P-(3/2,3/2)o", Null, 1559.706, 223.157, 5, 3, 
    Null, 2.2384*^8, 0., 0.048981}, 
   {"Si", "I", "", "3P-(3/2,3/2)o", Null, 1556.161, 77.115, 3, 3, 
    Null, 2.1873*^8, 0., 0.079411}, 
   {"Si", "I", "", "3P-(3/2,3/2)o", Null, 1555.66, 77.115, 3, 1, 
    Null, 5.3372*^8, 0., 0.064547}, 
   {"Si", "I", "", "3P-(3/2,3/2)o", Null, 1554.296, 0., 1, 3, Null, 
    7.0624*^7, 0., 0.076736}, 
   {"Si", "I", "", "3P-(1/2,3/2)o", Null, 1552.948, 0., 1, 3, Null, 
    1.1904*^7, 0., 0.012912}, 
   {"Si", "I", "", "3P-(1/2,3/2)o", Null, 1547.13, 0., 1, 3, Null, 
    8.4522*^6, 0., 0.0090991}, 
   {"Si", "I", "", "3P-(1/2,3/2)o", Null, 1542.432, 0., 1, 3, Null, 
    5.9239*^6, 0., 0.0063387}, 
   {"Si", "I41", ".12Au", "3P -  3p3 3So", Null, 1257.6348, 149.681, 
    9, 3, 1., 2.78*^9, 0., 0.21973}, 
   {"Si", "I41", ".12Au", "3P -  3p3 3So", Null, 1258.798, 223.157, 5, 3, 
    0.55556, 1.5402*^9, 0., 0.21953}, 
   {"Si", "I41", ".12Au", "3P -  3p3 3So", Null, 1256.488, 
    77.115, 3, 3, 0.33333, 9.292*^8, 0., 0.21993}, 
   {"Si", "I41", ".12Au", "3P -  3p3 3So", Null, 1255.272, 0., 1, 
    3, 0.11111, 3.1063*^8, 0., 0.22014}, 
   {"Si", "II", "0.01", "2Po-  3p2 4P ", 2350.173, 
    2350.892, 287.24, 4, 2, Null, 3000., 
    7550.99, 1.2428*^-6}, 
   {"Si", "II", "0.01", "2Po-  3p2 4P ", 2344.202, 2344.92, 
    287.24, 4, 4, Null, 1620.99, 1633.2, 
    1.3355*^-6}, {"Si", "II", "0.01", "2Po-  3p2 4P ", 2334.605, 
    2335.321, 287.24, 4, 6, Null, 2399.999999999999, 
    2399.999999999999, 2.9434*^-6}, 
   {"Si", "II", "0.01", "2Po-  3p2 4P ", 2334.4072, 2335.123, 0., 2, 2, 
    Null, 4550.99, 7550.99, 3.7195*^-6}, 
   {"Si", "II", "0.01", "2Po-  3p2 4P ", 2328.5163, 2329.231, 0., 2, 
    4, Null, 13.2, 1633.2, 2.1473*^-8}, 
   {"Si", "II", "1", "2Po-  3p2 2D ", Null, 1813.9815, 191.493, 6, 10, 1., 
    6.7*^6, 0., 0.0055087}, 
   {"Si", "II", "1", "2Po-  3p2 2D ", Null, 1817.4512, 287.24, 4, 
    4, 0.066679, 1.1103*^6, 6.7491*^6, 0.00054984}, 
   {"Si", "II", "1", "2Po-  3p2 2D ", Null, 1816.9285, 
    287.24, 4, 6, 0.6, 6.6675*^6, 
    6.6675*^6, 0.0049498}, 
   {"Si", "II", "1", "2Po-  3p2 2D ", Null, 1808.0126, 0., 2, 4, 
    0.33333, 5.6388*^6, 6.7491*^6, 0.0055268}, 
   {"Si", "II", "2", "2Po-  4s  2S ", Null, 1531.1831, 191.493, 6, 2, 1., 
    1.96*^9, 0., 0.22964}, {"Si", "II", "2", "2Po-  4s  2S ", Null, 
    1533.4312, 287.24, 4, 2, 0.66667, 
    1.3009*^9, 1.96*^9, 0.2293}, 
   {"Si", "II", "2", "2Po-  4s  2S ", Null, 1526.7066, 0., 2, 2, 
    0.33333, 6.5909*^8, 1.96*^9, 0.23031}, 
   {"Si", "II", "3", "2Po-  3p2 2S ", Null, 1307.6365, 191.493, 6, 2, 1., 
    1.72*^9, 0., 0.14697}, 
   {"Si", "II", "3", "2Po-  3p2 2S ", Null, 1309.2757, 287.24, 4, 
    2, 0.66667, 1.1424*^9, 1.72*^9, 0.14679}, 
   {"Si", "II", "3", "2Po-  3p2 2S ", Null, 1304.3702, 0., 2, 2, 0.33333, 
    5.7765*^8, 1.72*^9, 0.14734}, 
   {"Si", "II", "4", "2Po-  3d  2D ", Null, 1263.3135, 191.493, 6, 
    10, 1., 2.52*^9, 0., 1.0049}, 
   {"Si", "II", "4", "2Po-  3d  2D ", Null, 1265.002, 
    287.24, 4, 4, 0.066679, 4.1834*^8, 
    2.5328*^9, 0.10036}, {"Si", "II", "4", "2Po-  3d  2D ", Null, 
    1264.7377, 287.24, 4, 6, 0.6, 2.5115*^9, 2.5115*^9, 
    0.9034}, {"Si", "II", "4", "2Po-  3d  2D ", Null, 1260.4221, 0., 
    2, 4, 0.33333, 2.1145*^9, 2.5328*^9, 1.0072}, 
   {"Si", "II", "5", "2Po-  3p2 2P ", Null, 1194.0964, 191.493, 6, 
    6, 1., 3.5*^9, 0., 0.74818}, 
   {"Si", "II", "5", "2Po-  3p2 2P ", Null, 1197.3938, 
    287.24, 4, 2, 0.11111, 1.157*^9, 
    3.4951*^9, 0.12435}, 
   {"Si", "II", "5", "2Po-  3p2 2P ", Null, 1194.5002, 
    287.24, 4, 4, 0.55556, 2.9137*^9, 3.5025*^9, 
    0.62327}, {"Si", "II", "5", "2Po-  3p2 2P ", Null, 1193.2897, 
    0., 2, 2, 0.22222, 2.338*^9, 3.4951*^9, 0.49912}, 
   {"Si", "II", "5", "2Po-  3p2 2P ", Null, 1190.4158, 0., 2, 4, 
    0.11111, 5.8875*^8, 3.5025*^9, 0.25016}, 
   {"Si", "II", "5.01", "2Po-  5s  2S ", Null, 1022.6978, 191.493, 
    6, 2, 1., 5.4*^8, 1.28*^8, 0.028224}, 
   {"Si", "II", "5.01", "2Po-  5s  2S ", Null, 1023.7002, 
    287.24, 4, 2, 0.66667, 3.5895*^8, 
    6.6899*^8, 0.028197}, 
   {"Si", "II", "5.01", "2Po-  5s  2S ", Null, 1020.6989, 0., 2, 2, 
    0.33333, 1.8106*^8, 6.6899*^8, 0.028279}, 
   {"Si", "II", "6", "2Po-  4d  2D ", Null, 991.7453, 191.493, 6, 10, 1., 
    5.4*^8, 1.25*^8, 0.13271}, 
   {"Si", "II", "6", "2Po-  4d  2D ", Null, 992.6956, 287.24, 4, 
    4, 0.066679, 8.9746*^7, 6.6739*^8, 
    0.013259}, {"Si", "II", "6", "2Po-  4d  2D ", Null, 992.6828, 
    287.24, 4, 6, 0.6, 5.3847*^8, 6.6347*^8, 0.11933}, 
   {"Si", "II", "6", "2Po-  4d  2D ", Null, 989.8731, 0., 2, 4, 
    0.33333, 4.5255*^8, 6.6739*^8, 0.13296}, 
   {"Si", "III", "1", "1S -  3p  3Po", Null, 1892.03, 0., 1, 3, 1., 
    16699.99999999999, 16699.99999999999, 0.000026888}, 
   {"Si", "III", "2", "1S -  3p  1Po", Null, 1206.5, 0., 1, 3, 1., 
    2.5599*^9, 2.5599*^9, 1.6694}, 
   {"Si", "IV", "1", "2S -  3p  2Po", Null, 1396.7471, 0., 2, 6, 1., 
    8.768*^8, 0., 0.76933}, 
   {"Si", "IV", "1", "2S -  3p  2Po", Null, 1402.77, 0., 2, 2, 
    0.33333, 8.6555*^8, 8.6555*^8, 0.25534}, 
   {"Si", "IV", "1", "2S -  3p  2Po", Null, 1393.755, 0., 2, 4, 
    0.66667, 8.8246*^8, 8.8246*^8, 0.51399}, 
   {"P", "I", "1", "4So-  4s  4P ", Null, 1779.678, 0., 4, 12, 1., 2.16*^8, 
    0., 0.30769}, {"P", "I", "1", "4So-  4s  4P ", Null, 1787.6481, 0., 4, 
    2, 0.16667, 2.1313*^8, 2.1313*^8, 0.051054}, 
   {"P", "I", "1", "4So-  4s  4P ", Null, 1782.8291, 0., 4, 4, 0.33333, 
    2.1485*^8, 2.1485*^8, 0.10238}, 
   {"P", "I", "1", "4So-  4s  4P ", Null, 1774.9487, 0., 4, 6, 0.5, 
    2.1773*^8, 2.1773*^8, 0.15426}, 
   {"P", "I", "", "4So-  4s  2P ", Null, 1727.8148, 0., 4, 2, Null, 
    150699.9999999999, 4.1715*^8, 0.000033724}, 
   {"P", "I", "", "4So-  4s  2P ", Null, 1718.9702, 0., 4, 4, Null, 
    270110., 4.1727*^8, 0.00011966}, 
   {"P", "I", "2", "4So-  3p4 4P ", Null, 1676.6527, 0., 4, 12, 1., 3.9*^7, 
    0., 0.049309}, {"P", "I", "2", "4So-  3p4 4P ", Null, 1679.6969, 
    0., 4, 6, 0.5, 3.8788*^7, 0., 0.02461}, 
   {"P", "I", "2", "4So-  3p4 4P ", Null, 1674.5953, 0., 4, 4, 0.33333, 
    3.9144*^7, 0., 0.016456}, 
   {"P", "I", "2", "4So-  3p4 4P ", Null, 1671.6713, 0., 4, 2, 
    0.16667, 3.935*^7, 0., 0.00824298}, 
   {"P", "I", "", "4So-  4s  2D ", Null, 1534.7724, 0., 4, 4, Null, Null, 
    Null, Null}, {"P", "I", "", "4So-  4s  2D ", Null, 1534.7515, 0., 4, 6, 
    Null, 5419.9, 0., 2.8704*^-6}, 
   {"P", "I", "", "4So-  3d  2P ", Null, 1472.5559, 0., 4, 4, Null, 
    99069.8, 0., 0.000032206}, 
   {"P", "I", "", "4So-  3d  2P ", Null, 1467.8553, 0., 4, 2, Null, 
    28824., 0., 4.6552*^-6}, 
   {"P", "I", "", "4So-  3d  4F ", Null, 1466.6372, 0., 4, 4, Null, 
    536340., 0., 0.00017296}, {"P", "I", "", "4So-  3d  4F ", Null, 
    1464.9407, 0., 4, 6, Null, 2913.1, 0., 1.4059*^-6}, 
   {"P", "I", "", "4So-  3d  4D ", Null, 1431.0649, 0., 4, 2, Null, 151980., 
    0., 0.000023331}, 
   {"P", "I", "", "4So-  3d  4D ", Null, 1430.7317, 0., 4, 4, Null, 
    26361., 0., 8.0898*^-6}, 
   {"P", "I", "", "4So-  3d  4D ", Null, 1430.1149, 0., 4, 6, Null, 
    569180., 0., 0.00026178}, 
   {"P", "I", "", "4So-  5s  4P ", Null, 1376.3223, 0., 4, 12, 1., 
    3.372*^7, 0., 0.028728}, 
   {"P", "I", "", "4So-  5s  4P ", Null, 1381.6353, 0., 4, 2, 
    0.18677, 3.7353*^7, 0., 0.0053449}, 
   {"P", "I", "", "4So-  5s  4P ", Null, 1377.9343, 0., 4, 4, 
    0.34685, 3.4964*^7, 0., 0.0099527}, 
   {"P", "I", "", "4So-  5s  4P ", Null, 1373.4905, 0., 4, 6, 0.46638, 
    3.1648*^7, 0., 0.013426}, {"P", "I", "", "4So-  3d  4P ", Null, 
    1380.0577, 0., 4, 12, 1., 1.6208*^8, 0., 0.13883}, 
   {"P", "I", "", "4So-  3d  4P ", Null, 1381.476, 0., 4, 6, 
    0.57002, 1.8421*^8, 0., 0.079057}, 
   {"P", "I", "", "4So-  3d  4P ", Null, 1379.4282, 0., 4, 4, 
    0.30567, 1.4883*^8, 0., 0.042456}, 
   {"P", "I", "", "4So-  3d  4P ", Null, 1377.073, 0., 4, 2, 
    0.12431, 1.2167*^8, 0., 0.017296}, 
   {"P", "I", "", "4So-  3d  2D ", Null, 1374.7252, 0., 4, 4, Null, 
    68494., 0., 0.000019406}, {"P", "I", "", "4So-  3d  2D ", Null, 
    1372.027, 0., 4, 6, Null, 324540., 0., 0.00013739}, 
   {"P", "I", "", "4So-  4d  2P ", Null, 1321.421, 0., 4, 4, Null, 
    2.2909*^6, 0., 0.000599718}, 
   {"P", "I", "", "4So-  4d  2P ", Null, 1319.022, 0., 4, 2, Null, 
    5.0421*^7, 0., 0.0065757}, 
   {"P", "I", "", "4So-  4d  2F ", Null, 1318.684, 0., 4, 6, Null, 
    1.8018*^7, 0., 0.0070469}, 
   {"P", "I", "", "4So-  4d  4F ", Null, 1316.665, 0., 4, 4, Null, 
    Null, Null, Null}, {"P", "I", "", "4So-  4d  4F ", Null, 
    1314.994, 0., 4, 6, Null, 3.6995*^7, 0., 0.014386}, 
   {"P", "I", "", "4So-  4d  4P ", Null, 1286.444, 0., 4, 6, Null, 
    2.1243*^7, 0., 0.0079057}, 
   {"P", "I", "", "4So-  4d  4P ", Null, 1283.881, 0., 4, 4, Null, 
    3.4279*^7, 0., 0.0084711}, 
   {"P", "I", "", "4So-  4d  4P ", Null, 1282.62, 0., 4, 2, Null, Null, 
    Null, Null}, {"P", "I", "", "4So-  6d  2P ", Null, 1230.356, 0., 4, 4, 
    Null, Null, Null, Null}, {"P", "I", "", "4So-  6d  2P ", Null, 
    1226.742, 0., 4, 2, Null, 5.6965*^7, 0., 
    0.0064269}, {"P", "I", "", "4S0-  7d  4F ", Null, 1218.385, 
    0., 4, 4, Null, Null, Null, Null}, 
   {"P", "I", "", "4S0-  7d  4F ", Null, 1217.843, 0., 4, 6, Null, 
    1.9716*^7, 0., 0.0065757}, 
   {"P", "II", "", "3P -  3p3 5So", 2210.335, 2211.024, 
    469.12, 5, 5, Null, 1420.99, 0., 1.0407*^-6}, 
   {"P", "II", "", "3P -  3p3 5So", 2195.566, 2196.252, 164.9, 3, 
    5, Null, 508., 0., 6.1226*^-7}, 
   {"P", "II", "1", "3P -  3p3 3Do", Null, 1539.1996, 315.588, 9, 15, 1., 
    1.28*^7, 0., 0.0075771}, 
   {"P", "II", "1", "3P -  3p3 3Do", Null, 1543.631, 469.12, 5, 3, 
    0.0055699, 352780., 0., 0.0000756148}, 
   {"P", "II", "1", "3P -  3p3 3Do", Null, 1543.133, 469.12, 5, 5, 
    0.08333, 3.1755*^6, 0., 0.0011336}, 
   {"P", "II", "1", "3P -  3p3 3Do", Null, 1542.304, 469.12, 5, 7, 
    0.46667, 1.2723*^7, 0., 0.006352}, 
   {"P", "II", "1", "3P -  3p3 3Do", Null, 1536.416, 164.9, 3, 3, 
    0.08333, 5.3622*^6, 0., 0.0018976}, 
   {"P", "II", "1", "3P -  3p3 3Do", Null, 1535.923, 164.9, 3, 5, 
    0.25, 9.6616*^6, 0., 0.005695}, 
   {"P", "II", "1", "3P -  3p3 3Do", Null, 1532.533, 0., 1, 3, 
    0.11111, 7.2042*^6, 0., 0.00761}, 
   {"P", "II", "2", "3P -  3p3 3Po", Null, 1307.6832, 315.588, 9, 9, 1., 
    6.7*^7, 0., 0.017177}, 
   {"P", "II", "2", "3P -  3p3 3Po", Null, 1310.703, 469.12, 5, 5, 
    0.41667, 4.9904*^7, 0., 0.012853}, 
   {"P", "II", "2", "3P -  3p3 3Po", Null, 1309.874, 469.12, 5, 3, 0.13889, 
    2.7777*^7, 0., 0.0042879}, 
   {"P", "II", "2", "3P -  3p3 3Po", Null, 1305.497, 164.9, 3, 5, 0.13889, 
    1.6834*^7, 0., 0.0071689}, 
   {"P", "II", "2", "3P -  3p3 3Po", Null, 1304.675, 164.9, 3, 3, 0.08333, 
    1.6865*^7, 0., 0.0043039}, 
   {"P", "II", "2", "3P -  3p3 3Po", Null, 1304.492, 164.9, 3, 1, 
    0.11111, 6.7492*^7, 0., 0.0057395}, 
   {"P", "II", "2", "3P -  3p3 3Po", Null, 1301.874, 0., 1, 3, 
    0.11111, 2.2633*^7, 0., 0.017253}, 
   {"P", "II", "", "3P -  3d  1Do", Null, 1294.648, 469.12, 5, 5, Null, 
    742799.9999999999, 0., 0.00018665}, 
   {"P", "II", "", "3P -  3d  1Do", Null, 1289.569, 164.9, 3, 5, 
    Null, 820890., 0., 0.0003411}, 
   {"P", "II", "3", "3P -  4s  3Po", Null, 1154.42, 315.588, 9, 9, 1., 
    1.18*^9, 0., 0.23576}, {"P", "II", "3", "3P -  4s  3Po", Null, 1159.086, 
    469.12, 5, 3, 0.13889, 4.8576*^8, 0., 0.058703}, 
   {"P", "II", "3", "3P -  4s  3Po", Null, 1156.97, 164.9, 3, 1, 
    0.11111, 1.1722*^9, 0., 0.0784128}, 
   {"P", "II", "3", "3P -  4s  3Po", Null, 1155.014, 164.9, 3, 3, 
    0.08333, 2.9453*^8, 0., 0.058907}, 
   {"P", "II", "3", "3P -  4s  3Po", Null, 1153.995, 469.12, 5, 5, 
    0.41667, 8.8599*^8, 0., 0.17688}, 
   {"P", "II", "3", "3P -  4s  3Po", Null, 1152.818, 0., 1, 3, 
    0.11111, 3.9497*^8, 0., 0.23608}, 
   {"P", "II", "3", "3P -  4s  3Po", Null, 1149.958, 164.9, 3, 5, 0.13889, 
    2.9845*^8, 0., 0.098614}, 
   {"P", "II", "", "3P -  3d  3Fo", Null, 1145.017, 469.12, 5, 5, 
    Null, 154010., 0., 0.000030271}, 
   {"P", "II", "", "3P -  3d  3Fo", Null, 1142.887, 469.12, 5, 7, 
    Null, 939810., 0., 0.00025765}, 
   {"P", "II", "", "3P -  3d  3Fo", Null, 1141.042, 164.9, 3, 5, 
    Null, 490420., 0., 0.00015954}, 
   {"P", "II", "", "3P -  4s  1Po", Null, 1130.913, 469.12, 5, 3, Null, 
    1.3495*^6, 0., 0.00015525}, 
   {"P", "II", "", "3P -  4s  1Po", Null, 1127.036, 164.9, 3, 3, 
    Null, 2.8389*^6, 0., 0.0005406}, 
   {"P", "II", "", "3P -  4s  1Po", Null, 1124.945, 0., 1, 3, Null, 
    2.2119*^6, 0., 0.0012589}, 
   {"P", "II", "", "3P -  3d  1Po", Null, 977.239, 469.12, 5, 3, 
    Null, 1.2219*^7, 0., 0.0010496}, 
   {"P", "II", "", "3P -  3d  1Po", Null, 974.342, 164.9, 3, 3, Null, 
    4.07*^6, 0., 0.00057927}, 
   {"P", "II", "", "3P -  3d  1Po", Null, 972.779, 0., 1, 3, Null, 
    3.0974*^6, 0., 0.0013183}, 
   {"P", "II", "", "3P -  3d  3Po", Null, 967.2053, 315.588, 9, 9, 1., 
    3.9992*^9, 0., 0.56087}, 
   {"P", "II", "", "3P -  3d  3Po", Null, 969.363, 469.12, 5, 5, 0.12684, 
    9.07*^8, 0., 0.12777}, 
   {"P", "II", "", "3P -  3d  3Po", Null, 968.178, 469.12, 5, 3, 
    0.054098, 6.47*^8, 0., 0.054553}, 
   {"P", "II", "", "3P -  3d  3Po", Null, 966.5128, 164.9, 3, 5, 
    0.42972, 3.1*^9, 0., 0.72357}, 
   {"P", "II", "", "3P -  3d  3Po", Null, 965.335, 164.9, 3, 3, 
    0.00013, 1.6*^6, 0., 0.00022353}, 
   {"P", "II", "", "3P -  3d  3Po", Null, 963.801, 0., 1, 3, 
    0.28783, 3.4999*^9, 0., 1.4581}, 
   {"P", "II", "", "3P -  3d  3Po", Null, 963.619, 164.9, 3, 1, 0.10138, 
    3.6999*^9, 0., 0.17123}, 
   {"P", "II", "", "3P -  3d  3Do", Null, 963.832, 315.588, 9, 15, 1., 
    5.3775*^9, 0., 1.2482}, 
   {"P", "II", "", "3P -  3d  3Do", Null, 965.427, 469.12, 5, 7, 0.48752, 
    5.5999*^9, 0., 1.0935}, 
   {"P", "II", "", "3P -  3d  3Do", Null, 965.394, 469.12, 5, 3, 0.03528, 
    9.44*^8, 0., 0.079139}, 
   {"P", "II", "", "3P -  3d  3Do", Null, 964.947, 469.12, 5, 5, 
    0.20029, 3.22*^9, 0., 0.44949}, 
   {"P", "II", "", "3P -  3d  3Do", Null, 962.567, 164.9, 3, 3, 
    0.12262, 3.31*^9, 0., 0.45978}, 
   {"P", "II", "", "3P -  3d  3Do", Null, 962.123, 164.9, 3, 5, 0.12332, 
    2.*^9, 0., 0.46259}, 
   {"P", "II", "", "3P -  3d  3Do", Null, 961.041, 0., 1, 3, 
    0.03097, 8.4*^8, 0., 0.34893}, 
   {"P", "II", "", "3P -  3d  1Fo", Null, 935.531, 469.12, 5, 7, 
    Null, 45387., 0., 8.3374*^-6}, 
   {"P", "III", "", "2Po-  3p2 4P ", Null, 1774.228, 559.14, 4, 2, Null, 
    22136., 56308., 5.2232*^-6}, 
   {"P", "III", "", "2Po-  3p2 4P ", Null, 1767.82, 559.14, 4, 4, 
    Null, 14362., 15793., 6.7288*^-6}, 
   {"P", "III", "", "2Po-  3p2 4P ", Null, 1757.628, 559.14, 4, 6, 
    Null, 43265., 43265., 0.000030057}, 
   {"P", "III", "", "2Po-  3p2 4P ", Null, 1756.8, 0., 2, 2, Null, 
    34172., 56308., 0.000015811}, 
   {"P", "III", "", "2Po-  3p2 4P ", Null, 1750.517, 0., 2, 4, 
    Null, 1431.4, 15793., 1.3151*^-6}, 
   {"P", "III", "1", "2Po-  3p2 2D ", Null, 1341.1746, 
    372.76, 6, 10, 1., 5.6999*^7, 0., 0.025169}, 
   {"P", "III", "1", "2Po-  3p2 2D ", Null, 1344.85, 559.14, 4, 4, 
    0.066679, 9.2575*^6, 5.6594*^7, 0.0025101}, 
   {"P", "III", "1", "2Po-  3p2 2D ", Null, 1344.326, 559.14, 4, 6, 0.6, 
    5.5607*^7, 5.5607*^7, 0.022599}, 
   {"P", "III", "1", "2Po-  3p2 2D ", Null, 1334.813, 0., 2, 4, 0.33333, 
    4.7337*^7, 5.6594*^7, 0.025289}, 
   {"P", "III", "2", "2Po-  3p2 2S ", Null, 1001.7263, 372.76, 6, 
    2, 1., 2.22*^9, 0., 0.11132}, 
   {"P", "III", "2", "2Po-  3p2 2S ", Null, 1003.6, 559.14, 4, 2, 
    0.66667, 1.4717*^9, 2.22*^9, 0.11112}, 
   {"P", "III", "2", "2Po-  3p2 2S ", Null, 998., 0., 2, 2, 0.33333, 
    7.4831*^8, 2.22*^9, 0.11174}, 
   {"P", "III", "3", "2Po-  3p2 2P ", Null, 918.1474, 
    372.76, 6, 6, 1., 4.8999*^9, 0., 0.60663}, 
   {"P", "III", "3", "2Po-  3p2 2P ", Null, 921.845, 559.14, 4, 2, 
    0.11111, 1.5808*^9, 4.7916*^9, 0.10079}, 
   {"P", "III", "3", "2Po-  3p2 2P ", Null, 918.663, 559.14, 4, 4, 
    0.55556, 3.9933*^9, 4.8043*^9, 0.505248}, 
   {"P", "III", "3", "2Po-  3p2 2P ", Null, 917.118, 0., 2, 2, 
    0.22222, 3.2108*^9, 4.7916*^9, 0.40487}, 
   {"P", "III", "3", "2Po-  3p2 2P ", Null, 913.968, 0., 2, 4, 
    0.11111, 8.1102*^8, 4.8043*^9, 
    0.20313}, {"P", "IV", "", "1S -  3p  3Po", Null, 
    1467.427, 0., 1, 3, 1., 62500., 62500., 0.000060539}, 
   {"P", "IV", "1", "1S -  3p  1Po", Null, 950.657, 0., 1, 3, 1., 
    2.86*^9, 2.86*^9, 1.1625}, {"P", "V", "1", "2S -  3p  2Po", Null, 
    1121.3008, 0., 2, 6, 1., 1.2514*^9, 0., 0.70765}, 
   {"P", "V", "1", "2S -  3p  2Po", Null, 1128.008, 0., 2, 2, 0.33333, 
    1.2292*^9, 1.2292*^9, 0.23448}, 
   {"P", "V", "1", "2S -  3p  2Po", Null, 1117.977, 0., 2, 4, 
    0.66667, 1.2626*^9, 1.2626*^9, 0.47317}, 
   {"S", "I", "1", "3P -  4s  5So", Null, 1914.697, 
    396.055, 3, 5, Null, 19000., 0., 0.000017404}, 
   {"S", "I", "1", "3P -  4s  5So", Null, 1900.2866, 0., 5, 5, Null, 
    65999.99999999999, 0., 0.000035739}, 
   {"S", "I", "2", "3P -  4s  3So", Null, 1813.7281, 195.756, 9, 3, 1., 
    6.7*^8, 0., 0.11014}, 
   {"S", "I", "2", "3P -  4s  3So", Null, 1826.2448, 
    573.64, 1, 3, 0.11111, 7.2924*^7, 6.7003*^8, 
    0.10939}, {"S", "I", "2", "3P -  4s  3So", Null, 1820.3412, 
    396.055, 3, 3, 0.33333, 2.2091*^8, 6.7003*^8, 
    0.10974}, {"S", "I", "2", "3P -  4s  3So", Null, 
    1807.3113, 0., 5, 3, 0.55556, 3.762*^8, 6.7003*^8, 
    0.11053}, {"S", "I", "3", "3P -  4s' 3Do", Null, 
    1478.5044, 195.756, 9, 15, 1., 1.6496*^8, 0., 
    0.090103}, {"S", "I", "3", "3P -  4s' 3Do", Null, 1487.15, 
    573.64, 1, 3, 0.10734, 8.7999*^7, 
    1.682*^8, 0.086538}, 
   {"S", "I", "3", "3P -  4s' 3Do", Null, 1483.2329, 
    396.055, 3, 3, 0.091899, 7.5*^7, 
    1.682*^8, 0.024736}, 
   {"S", "I", "3", "3P -  4s' 3Do", Null, 1483.0385, 396.055, 3, 
    5, 0.24472, 1.2*^8, 1.7*^8, 0.065946}, 
   {"S", "I", "3", "3P -  4s' 3Do", Null, 1474.5706, 0., 5, 3, 
    0.0074699, 6.2*^6, 1.682*^8, 0.0012126}\
, {"S", "I", "3", 
      "3P -  4s' 3Do", Null, 1474.3785, 0., 5, 5, 
    0.10019, 5.*^7, 1.7*^8, 0.016295}, 
   {"S", "I", "3", "3P -  4s' 3Do", Null, 1473.9943, 0., 5, 7, 0.4485, 
    1.6*^8, 1.6*^8, 0.0729628}, 
   {"S", "I", "4", "3P -  3d  5Do", Null, 1485.6215, 573.64, 1, 
    3, Null, 2.3999*^6, 0., 0.0022831}, 
   {"S", "I", "4", "3P -  3d  5Do", Null, 1481.7426, 396.055, 3, 
    1, Null, Null, Null, Null}, 
   {"S", "I", "4", "3P -  3d  5Do", Null, 1481.7124, 
    396.055, 3, 3, Null, Null, Null, Null}, 
   {"S", "I", "4", "3P -  3d  5Do", Null, 1481.6626, 396.055, 3, 
    5, Null, 1.7*^7, 0., 0.0093251}, 
   {"S", "I", "4", "3P -  3d  5Do", Null, 1473.0679, 0., 5, 3, Null, Null, 
    Null, Null}, {"S", "I", "4", "3P -  3d  5Do", Null, 1473.0186, 0., 5, 5, 
    Null, Null, Null, Null}, {"S", "I", "4", "3P -  3d  5Do", Null, 
    1472.9706, 0., 5, 7, Null, 4.2*^7, 0., 0.019126}, 
   {"S", "I", "", "3P -  4s' 1Do", Null, 1452.6052, 
    396.055, 3, 5, Null, 350000., 0., 0.00018453}, 
   {"S", "I", "", "3P -  4s' 1Do", Null, 1444.296, 0., 5, 5, Null, 2.6*^6, 
    0., 0.0008131}, {"S", "I", "5", "3P -  3d  3Do", Null, 
    1429.1072, 195.756, 9, 15, 1., 4.4856*^8, 0., 0.2289}, 
   {"S", "I", "5", "3P -  3d  3Do", Null, 1436.9671, 
    573.64, 1, 3, 0.10879, 2.4999*^8, 0., 
    0.22289}, {"S", "I", "5", "3P -  3d  3Do", Null, 1433.3095, 
    396.055, 3, 3, 0.08547, 1.9*^8, 0., 0.058518}, 
   {"S", "I", "5", "3P -  3d  3Do", Null, 1433.2779, 
    396.055, 3, 5, 0.24738, 3.3999*^8, 0., 
    0.16939}, {"S", "I", "5", "3P -  3d  3Do", Null, 1425.2189, 
    0., 5, 3, 0.00575, 1.3*^7, 0., 0.0023753}, 
   {"S", "I", "5", "3P -  3d  3Do", Null, 1425.1877, 0., 5, 5, 
    0.08844, 1.2*^8, 0., 0.036541}, 
   {"S", "I", "5", "3P -  3d  3Do", Null, 1425.0299, 0., 5, 7, 0.46417, 
    4.5*^8, 0., 0.19189}, 
   {"S", "I", "", "3P -  5s  5So", Null, 1422.3388, 
    396.055, 3, 5, Null, Null, Null, Null}, 
   {"S", "I", "", "3P -  5s  5So", Null, 1414.3713, 0., 5, 5, Null, 
    Null, Null, Null}, {"S", "I", "6", "3P -  5s  3So", Null, 
    1405.3699, 195.756, 9, 3, 1., 1.5694*^8, 0., 
    0.015499}, {"S", "I", "6", "3P -  5s  3So", Null, 
    1412.8732, 573.64, 1, 3, 0.10359, 1.6*^7, 0., 
    0.014365}, {"S", "I", "6", "3P -  5s  3So", Null, 1409.3371, 
    396.055, 3, 3, 0.32131, 5.*^7, 0., 
    0.014889}, {"S", "I", "6", "3P -  5s  3So", Null, 1401.5142, 
    0., 5, 3, 0.5751, 9.1*^7, 0., 
    0.016078}, {"S", "I", "7", "3P -  3p5 3Po", Null, 1388.7297, 
    195.756, 9, 9, 1., 1.1793*^6, 0., 0.00034096}, 
   {"S", "I", "7", "3P -  3p5 3Po", Null, 1396.113, 396.055, 3, 5, 
    0.27283, 570000., 0., 0.0002776}, 
   {"S", "I", "7", "3P -  3p5 3Po", Null, 1392.5892, 
    573.64, 1, 3, 0.14821, 520000., 0., 
    0.000453558}, {"S", "I", "7", "3P -  3p5 3Po", Null, 
    1389.1537, 396.055, 3, 3, 0.0481, 170000., 0., 
    0.000049182}, {"S", "I", "7", "3P -  3p5 3Po", Null, 1388.4358, 0., 5, 
    5, 0.25894, 550000., 0., 0.00015895}, 
   {"S", "I", "7", "3P -  3p5 3Po", Null, 1385.5103, 
    396.055, 3, 1, 0.12164, 1.3*^6, 0., 0.00012471}, 
   {"S", "I", "7", "3P -  3p5 3Po", Null, 1381.5527, 0., 5, 3, 
    0.15028, 540000., 0., 0.000092712}, 
   {"S", "I", "8", "3P -  4d  3Do", Null, 1319.9845, 195.756, 9, 15, 
    1., 9.4*^7, 0., 0.040923}, 
   {"S", "I", "8", "3P -  4d  3Do", Null, 1326.6416, 
    573.64, 1, 3, 0.11111, 5.1439*^7, 0., 
    0.040718}, {"S", "I", "8", "3P -  4d  3Do", Null, 
    1323.5235, 396.055, 3, 3, 0.08333, 
    3.8852*^7, 0., 0.010203}, 
   {"S", "I", "8", "3P -  4d  3Do", Null, 1323.5165, 
    396.055, 3, 5, 0.25, 6.9937*^7, 0., 0.030611}, 
   {"S", "I", "8", "3P -  4d  3Do", Null, 1316.6219, 0., 5, 3, 
    0.0055699, 2.6333*^6, 0., 0.00041061}, 
   {"S", "I", "8", "3P -  4d  3Do", Null, 1316.615, 0., 5, 5, 
    0.08333, 2.3689*^7, 0., 0.0061549}, 
   {"S", "I", "8", "3P -  4d  3Do", Null, 1316.5425, 0., 5, 7, 
    0.46667, 9.474*^7, 0., 0.034466}, 
   {"S", "I", "", "3P -  6s  3So", Null, 1306.7643, 195.756, 9, 3, 
    1., 3.4999*^8, 0., 0.029014}, 
   {"S", "I", "", "3P -  6s  3So", Null, 1313.2492, 573.64, 1, 3, 
    0.11111, 3.7221*^7, 0., 0.028871}, 
   {"S", "I", "", "3P -  6s  3So", Null, 1310.1937, 396.055, 3, 3, 
    0.33333, 1.1244*^8, 0., 0.028938}, 
   {"S", "I", "", "3P -  6s  3So", Null, 1303.43, 0., 5, 3, 0.55556, 
    1.9034*^8, 0., 0.029089}, 
   {"S", "I", "9", "3P -  4s''3Po", Null, 1299.2077, 195.756, 9, 9, 
    1., 6.6911*^8, 0., 0.16932}, 
   {"S", "I", "9", "3P -  4s''3Po", Null, 1305.8837, 573.64, 1, 
    3, 0.12141, 2.4999*^8, 0., 0.18408}, 
   {"S", "I", "9", "3P -  4s''3Po", Null, 1303.1105, 396.055, 3, 
    1, 0.11059, 6.6999*^8, 0., 0.056007}, 
   {"S", "I", "9", "3P -  4s''3Po", Null, 1302.8623, 
    396.055, 3, 3, 0.08038, 1.6*^8, 0., 0.040717}, 
   {"S", "I", "9", "3P -  4s''3Po", Null, 1302.3361, 
    396.055, 3, 5, 0.15053, 1.8*^8, 0., 0.076282}, 
   {"S", "I", "9", "3P -  4s''3Po", Null, 1296.1739, 0., 5, 3, 
    0.13357, 2.7*^8, 0., 0.040804}, 
   {"S", "I", "9", "3P -  4s''3Po", Null, 1295.6531, 0., 5, 5, 
    0.40351, 4.9*^8, 0., 0.12332}, 
   {"S", "I", "", "3P -  4s\" 1Po", Null, 1286.7562, 573.64, 1, 
    3, Null, 1729.9, 0., 1.2882*^-6}, 
   {"S", "I", "", "3P -  4s\" 1Po", Null, 1283.8225, 
    396.055, 3, 3, Null, 77.627, 0., 
    1.9181*^-8}, {"S", "I", "", "3P -  4s\" 1Po", Null, 
    1277.3278, 0., 5, 3, Null, 121460., 0., 0.000017825}, 
   {"S", "I", "", "3P -  5d  3Do", Null, 1273.9495, 195.756, 9, 15, 1., 
    1.6247*^8, 0., 0.065884}, 
   {"S", "I", "", "3P -  5d  3Do", Null, 1280.1001, 573.64, 1, 3, 
    0.11196, 8.9647*^7, 0., 0.066069}, 
   {"S", "I", "", "3P -  5d  3Do", Null, 1277.2156, 
    396.055, 3, 5, 0.25008, 1.2096*^8, 0., 0.0493048}, 
   {"S", "I", "", "3P -  5d  3Do", Null, 1277.1967, 
    396.055, 3, 3, 0.08474, 6.8314*^7, 0., 0.016706}, 
   {"S", "I", "", "3P -  5d  3Do", Null, 1270.7874, 0., 5, 5, 
    0.08431, 4.1403*^7, 0., 0.010024}, 
   {"S", "I", "", "3P -  5d  3Do", Null, 1270.7804, 0., 5, 7, 0.46333, 
    1.6252*^8, 0., 0.055085}, 
   {"S", "I", "", "3P -  5d  3Do", Null, 1270.7686, 0., 5, 3, 
    0.0055799, 4.5592*^6, 0., 0.00066226}, 
   {"S", "I", "", "3P -  7s  3So", Null, 1265.9895, 195.756, 9, 3, 
    Null, Null, Null, Null}, {"S", "I", "", "3P -  7s  3So", Null, 
    1272.0751, 573.64, 1, 3, Null, Null, Null, Null}, 
   {"S", "I", "", "3P -  7s  3So", Null, 1269.208, 396.055, 3, 3, 
    Null, Null, Null, Null}, {"S", "I", "", "3P -  7s  3So", Null, 
    1262.8599, 0., 5, 3, Null, Null, Null, Null}, 
   {"S", "I", "", "3P -  6d  3Do", Null, 1256.0927, 
    573.64, 1, 3, Null, 5.3577*^7, 0., 
    0.038019}, {"S", "I", "", "3P -  6d  3Do", Null, 1253.3248, 
    396.055, 3, 5, Null, 7.39688*^7, 0., 
    0.029032}, {"S", "I", "", "3P -  6d  3Do", Null, 1253.297, 
    396.055, 3, 3, Null, 4.0824*^7, 0., 0.0096134}, 
   {"S", "I", "", "3P -  6d  3Do", Null, 1247.1602, 0., 5, 7, Null, 
    9.9358*^7, 0., 0.032436}, 
   {"S", "I", "", "3P -  6d  3Do", Null, 1247.1342, 0., 5, 5, Null, 
    2.4737*^7, 0., 0.0057681}, 
   {"S", "I", "", "3P -  6d  3Do", Null, 1247.1067, 0., 5, 3, Null, 
    2.7249*^6, 0., 0.00038109}, 
   {"S", "I", "", "3P -  8s  3So", Null, 1241.905, 0., 5, 3, Null, Null, 
    Null, Null}, {"S", "I", "", "3P -  7d  3Do", Null, 1236.3059, 195.756, 
    9, 15, 1., 6.1871*^7, 0., 0.023629}, 
   {"S", "I", "", "3P -  7d  3Do", Null, 1242.0658, 
    573.64, 1, 3, 0.11075, 3.3786*^7, 0., 
    0.023442}, {"S", "I", "", "3P -  7d  3Do", Null, 
    1239.3662, 396.055, 3, 5, 0.24739, 
    4.5589*^7, 0., 0.017494}, 
   {"S", "I", "", "3P -  7d  3Do", Null, 1239.3322, 396.055, 3, 3, 
    0.08383, 2.5742*^7, 0., 0.0059276}, 
   {"S", "I", "", "3P -  7d  3Do", Null, 1233.3454, 0., 5, 7, 
    0.46911, 6.2643*^7, 0., 0.02}, 
   {"S", "I", "", "3P -  7d  3Do", Null, 1233.3124, 0., 5, 5, 
    0.083428, 1.5596*^7, 0., 0.0035566}, 
   {"S", "I", "", "3P -  7d  3Do", Null, 1233.2787, 0., 5, 3, 0.00551, 
    1.7175*^6, 0., 0.00023498}, 
   {"S", "I", "", "3P -  8d  3Do", Null, 1227.4589, 195.756, 9, 15, 1., 
    3.9105*^7, 0., 0.014721}, {"S", "I", "", "3P -  8d  3Do", Null, 
    1233.1324, 573.64, 1, 3, 0.11215, 
    2.1627*^7, 0., 0.014791}, {"S", "I", "", "3P -  8d  3Do", Null, 
    1230.4731, 396.055, 3, 5, 0.25054, 
    2.9176*^7, 0., 0.011038}, 
   {"S", "I", "", "3P -  8d  3Do", Null, 1230.438, 
    396.055, 3, 3, 0.08489, 1.6478*^7, 0., 0.0037401}, 
   {"S", "I", "", "3P -  8d  3Do", Null, 1224.544, 0., 5, 7, 0.46428, 
    3.9183*^7, 0., 0.012332}, 
   {"S", "I", "", "3P -  8d  3Do", Null, 1224.5056, 0., 5, 5, 
    0.082568, 9.7555*^6, 0., 0.002193}, 
   {"S", "I", "", "3P -  8d  3Do", Null, 1224.4709, 0., 5, 3, 0.00558, 
    1.0993*^6, 0., 0.00014826}, 
   {"S", "I", "", "3P -  9d  3Do", Null, 1218.5953, 0., 5, 7, Null, Null, 
    Null, Null}, {"S", "I", "", "3P -  9d  3Do", Null, 1218.57, 0., 5, 5, 
    Null, Null, Null, Null}, {"S", "I", "", "3P -  9d  3Do", Null, 
    1218.5133, 0., 5, 3, Null, Null, Null, Null}, 
   {"S", "I", "", "3P - 10d  3Do", Null, 1214.3177, 0., 5, 7, Null, Null, 
    Null, Null}, {"S", "I", "", "3P - 10d  3Do", Null, 1214.2944, 
    0., 5, 5, Null, Null, Null, Null}, 
   {"S", "I", "", "3P - 10d  3Do", Null, 1214.2812, 0., 5, 3, Null, 
    Null, Null, Null}, {"S", "I", "", "3P - 11d  3Do", Null, 1211.212, 0., 
    5, 7, Null, Null, Null, Null}, 
   {"S", "I", "", "3P - 11d  3Do", Null, 1211.187, 0., 5, 5, Null, Null, 
    Null, Null}, {"S", "I", "", "3P - 11d  3Do", Null, 1211.176, 0., 5, 3, 
    Null, Null, Null, Null}, {"S", "I", "", "3P - 12d  3Do", Null, 1208.851, 
    0., 5, 7, Null, Null, Null, Null}, 
   {"S", "I", "", "3P - 12d  3Do", Null, 1208.824, 0., 5, 5, Null, Null, 
    Null, Null}, {"S", "I", "", "3P - 13d  3Do", Null, 1207.015, 0., 5, 7, 
    Null, Null, Null, Null}, {"S", "I", "", "3P - 13d  3Do", Null, 
    1206.996, 0., 5, 5, Null, Null, Null, Null}, 
   {"S", "I", "", "3P - 14d  3Do", Null, 1205.564, 0., 5, 7, Null, Null, 
    Null, Null}, {"S", "II", "1", "4So-  3p4 4P ", Null, 1256.1171, 
    0., 4, 12, 1., 4.5999*^7, 0., 0.032572}, 
   {"S", "II", "1", "4So-  3p4 4P ", Null, 1259.519, 0., 4, 6, 0.5, 
    4.5529*^7, 4.5699*^7, 0.016242}, 
   {"S", "II", "1", "4So-  3p4 4P ", Null, 1253.811, 0., 4, 4, 0.33333, 
    4.6153*^7, 4.62*^7, 0.010877}, 
   {"S", "II", "1", "4So-  3p4 4P ", Null, 1250.584, 0., 4, 2, 0.16667, 
    4.6513*^7, 5.2699*^7, 0.0054529}, 
   {"S", "II", "", "4So-  3p4 2D ", Null, 1021.547, 0., 4, 4, Null, 
    Null, Null, Null}, {"S", "II", "", "4So-  3p4 2D ", Null, 1021.254, 0., 
    4, 6, Null, Null, Null, Null}, 
   {"S", "II", "", "4So-  3d  2P ", Null, 946.978, 0., 4, 4, Null, Null, 
    Null, Null}, {"S", "II", "", "4So-  3d  2P ", Null, 943.003, 0., 2, 4, 
    Null, Null, Null, Null}, {"S", "II", "2", "4So-  4s  4P ", Null, 
    909.0498, 0., 4, 12, 1., 1.13*^9, 0., 0.41998}, 
   {"S", "II", "2", "4So-  4s  4P ", Null, 912.736, 0., 4, 2, 
    0.16667, 1.1164*^9, 0., 0.069716}, 
   {"S", "II", "2", "4So-  4s  4P ", Null, 910.485, 0., 4, 4, 0.33333, 
    1.1247*^9, 0., 0.13977}, 
   {"S", "II", "2", "4So-  4s  4P ", Null, 906.876, 0., 4, 6, 0.5, 
    1.1381*^9, 0., 0.21049}, 
   {"S", "III", "", "3P -  3p3 5So", Null, 1728.942, 
    833.08, 5, 5, Null, 7320., 0., 3.2804*^-6}, 
   {"S", "III", "", "3P -  3p3 5So", Null, 1713.114, 
    298.69, 3, 5, Null, 2660., 0., 1.9506*^-6}, 
   {"S", "III", "1", "3P -  3p3 3Do", Null, 1197.5596, 562.385, 9, 
    15, 1., 6.15*^7, 0., 0.022038}, 
   {"S", "III", "1", "3P -  3p3 3Do", Null, 1202.128, 
    833.08, 5, 3, 0.0055699, 1.6903*^6, 
    6.2318*^7, 0.00021972}, 
   {"S", "III", "1", "3P -  3p3 3Do", Null, 1201.729, 
    833.08, 5, 5, 0.08333, 1.5215*^7, 6.1747*^7, 
    0.0032941}, {"S", "III", "1", "3P -  3p3 3Do", Null, 1200.97, 
    833.08, 5, 7, 0.46667, 6.0978*^7, 
    6.0978*^7, 0.01846}, 
   {"S", "III", "1", "3P -  3p3 3Do", Null, 1194.455, 298.69, 3, 
    3, 0.08333, 2.5824*^7, 6.2318*^7, 0.0055236}, 
   {"S", "III", "1", "3P -  3p3 3Do", Null, 1194.061, 298.69, 3, 
    5, 0.25, 4.6532*^7, 6.1747*^7, 0.016577}, 
   {"S", "III", "1", "3P -  3p3 3Do", Null, 1190.208, 0., 1, 3, 
    0.11111, 3.4803*^7, 6.2318*^7, 0.022174}\
, {"S", "III", "2", 
      "3P -  3p3 3Po", Null, 1018.4108, 562.385, 9, 9, 1., 
    2.27*^8, 0., 0.035296}, 
   {"S", "III", "2", "3P -  3p3 3Po", Null, 1021.328, 833.08, 5, 
    5, 0.41667, 1.688*^8, 2.2599*^8, 0.026397}, 
   {"S", "III", "2", "3P -  3p3 3Po", Null, 1021.115, 
    833.08, 5, 3, 0.13889, 9.3835*^7, 2.2806*^8, 
    0.00880088}, {"S", "III", "2", "3P -  3p3 3Po", Null, 
    1015.784, 298.69, 3, 5, 0.13889, 5.71928*^7, 
    2.2599*^8, 0.014745}, 
   {"S", "III", "2", "3P -  3p3 3Po", Null, 1015.574, 
    298.69, 3, 3, 0.08333, 5.7225*^7, 2.2806*^8, 
    0.0088484}, {"S", "III", "2", "3P -  3p3 3Po", Null, 
    1015.509, 298.69, 3, 1, 0.11111, 
    2.2895*^8, 2.2895*^8, 0.011799}, 
   {"S", "III", "2", "3P -  3p3 3Po", Null, 1012.502, 0., 1, 3, 
    0.11111, 7.6998*^7, 2.2806*^8, 0.035502}, 
   {"S", "IV", "", "2Po-  3p2 4P ", Null, 1423.83, 951.43, 4, 2, 
    Null, 16840.9, 43790.9, 2.5591*^-6}, 
   {"S", "IV", "", "2Po-  3p2 4P ", Null, 1416.9, 951.43, 4, 4, Null, 
    10970.9, 11613., 3.3017*^-6}, 
   {"S", "IV", "", "2Po-  3p2 4P ", Null, 1406.02, 951.43, 4, 6, Null, 
    27299.99999999999, 27299.99999999999, 0.000012137}, 
   {"S", "IV", "", "2Po-  3p2 4P ", Null, 1404.8, 0., 2, 2, Null, 
    26950.9, 43790.9, 7.9734*^-6}, 
   {"S", "IV", "", "2Po-  3p2 4P ", Null, 1398.05, 0., 2, 4, Null, 
    642.59, 11613., 3.7653*^-7}, 
   {"S", "IV", "1", "2Po-  3p2 2D ", Null, 1069.5505, 634.286, 6, 
    10, 1., 1.39*^8, 0., 0.039739}, 
   {"S", "IV", "1", "2Po-  3p2 2D ", Null, 1073.516, 951.43, 4, 4, 
    0.066679, 2.2912*^7, 1.4101*^8, 
    0.0039586}, {"S", "IV", "1", "2Po-  3p2 2D ", Null, 1072.974, 
    951.43, 4, 6, 0.6, 1.3767*^8, 1.3767*^8, 0.035643}, 
   {"S", "IV", "1", "2Po-  3p2 2D ", Null, 1062.662, 0., 2, 4, 
    0.33333, 1.181*^8, 1.4101*^8, 0.039988}, 
   {"S", "V", "", "1S -  3p  3Po", Null, 1199.134, 0., 1, 3, 1., 
    163999.9999999999, 163999.9999999999, 0.00010606}, 
   {"S", "VI", "1", "2S -  3p  2Po", Null, 937.0637, 0., 2, 6, 1., 
    1.6799*^9, 0., 0.65953}, 
   {"S", "VI", "1", "2S -  3p  2Po", Null, 944.523, 0., 2, 2, 0.33333, 
    1.6307*^9, 1.6307*^9, 0.2181}, 
   {"S", "VI", "1", "2S -  3p  2Po", Null, 933.378, 0., 2, 4, 
    0.66667, 1.6899*^9, 1.6899*^9, 0.44142}, 
   {"Cl", "I", "1", "2Po-  4s  4P ", Null, 1396.527, 882.36, 2, 4, Null, 
    1.2793*^6, 1.1225*^7, 0.00074812}, 
   {"Cl", "I", "1", "2Po-  4s  4P ", Null, 1389.9569, 882.36, 2, 2, Null, 
    1.9147*^6, 2.4008*^6, 0.00055459}, 
   {"Cl", "I", "1", "2Po-  4s  4P ", Null, 1389.6925, 0., 4, 6, Null, 
    250120., 250120., 0.00010863}, 
   {"Cl", "I", "1", "2Po-  4s  4P ", Null, 1379.5279, 0., 4, 4, 
    Null, 9.9454*^6, 1.1225*^7, 0.0028375}, 
   {"Cl", "I", "1", "2Po-  4s  4P ", Null, 1373.1164, 0., 4, 2, Null, 
    486070., 2.4008*^6, 0.000068697}, 
   {"Cl", "I", "2", "2Po-  4s  2P ", Null, 1348.7086, 
    294.12, 6, 6, 1., 5.1967*^8, 0., 0.14172}, 
   {"Cl", "I", "2", "2Po-  4s  2P ", Null, 1363.4476, 882.36, 2, 4, 
    0.096869, 7.30818*^7, 5.0879*^8, 0.040735}, 
   {"Cl", "I", "2", "2Po-  4s  2P ", Null, 1351.6563, 882.36, 2, 2, 
    0.23034, 3.5676*^8, 5.4241*^8, 0.097717}\
, {"Cl", "I", "2", 
      "2Po-  4s  2P ", Null, 1347.2396, 0., 4, 4, 
    0.55713, 4.3571*^8, 5.0879*^8, 0.11856}, 
   {"Cl", "I", "2", "2Po-  4s  2P ", Null, 1335.7258, 0., 4, 2, 
    0.11567, 1.8564*^8, 5.4241*^8, 0.024828}, 
   {"Cl", "I", "", "2Po-  4s' 2D ", Null, 1192.9361, 
    294.12, 6, 10, 1., 2.3011*^8, 0., 
    0.081823}, {"Cl", "I", "", "2Po-  4s' 2D ", Null, 
    1201.3526, 882.36, 2, 4, 0.36478, 2.0547*^8, 
    2.3128*^8, 0.088914}, {"Cl", "I", "", "2Po-  4s' 2D ", Null, 
    1188.7742, 0., 4, 6, 0.59082, 2.2898*^8, 2.2898*^8, 
    0.072768}, {"Cl", "I", "", "2Po-  4s' 2D ", Null, 
    1188.7515, 0., 4, 4, 0.04441, 2.5817*^7, 2.3128*^8, 
    0.0054694}, {"Cl", "I", "", "2Po-  3p6 2S ", Null, 1171.1683, 
    294.12, 6, 2, 1., 1.4083*^8, 0., 
    0.00965357}, {"Cl", "I", "", "2Po-  3p6 2S ", Null, 1179.2928, 
    882.36, 2, 2, 0.34132, 4.7084*^7, 0., 0.0098168}, 
   {"Cl", "I", "", "2Po-  3p6 2S ", Null, 1167.148, 0., 4, 2, 
    0.65868, 9.3726*^7, 0., 0.0095706}, 
   {"Cl", "I", "", "2Po-  3d  4D ", Null, 1145.3942, 882.36, 2, 4, 
    Null, 56127., 0., 0.000022079}, 
   {"Cl", "I", "", "2Po-  3d  4D ", Null, 1144.291, 882.36, 2, 2, 
    Null, 295150., 0., 0.0000579398}, 
   {"Cl", "I", "", "2Po-  3d  4D ", Null, 1135.331, 0., 4, 6, Null, 
    72736., 0., 0.000021083}, {"Cl", "I", "", "2Po-  3d  4D ", Null, 
    1133.9341, 0., 4, 4, Null, 270340., 0., 
    0.000052112}, {"Cl", "I", "", "2Po-  3d  4D ", Null, 
    1132.8528, 0., 4, 2, Null, 146480., 0., 0.000014091}, 
   {"Cl", "I", "", "2Po-  3d  4F ", Null, 1110.2948, 882.36, 2, 4, 
    Null, 6.2403*^6, 0., 0.0023066}, 
   {"Cl", "I", "", "2Po-  3d  4F ", Null, 1101.9361, 0., 4, 6, Null, 
    1.405*^8, 0., 0.038365}, {"Cl", "I", "", "2Po-  3d  4F ", Null, 
    1099.523, 0., 4, 4, Null, 2.1022*^7, 0., 0.0038101}\
, {"Cl", "I", "", 
      "2Po-  3d  4P ", Null, 1108.8114, 882.36, 2, 2, Null, 
    2.5906*^6, 0., 0.0004775}, 
   {"Cl", "I", "", "2Po-  3d  4P ", Null, 1103.0692, 882.36, 2, 4, 
    Null, 7.7961*^7, 0., 0.028443}, 
   {"Cl", "I", "", "2Po-  3d  4P ", Null, 1098.0683, 0., 4, 2, Null, 
    2.837*^6, 0., 0.00025641}, 
   {"Cl", "I", "", "2Po-  3d  4P ", Null, 1092.4365, 0., 4, 4, Null, 
    5.2034*^7, 0., 0.0093098}, {"Cl", "I", "", "2Po-  3d  4P ", Null, 
    1090.9815, 0., 4, 6, Null, 9.1066*^6, 0., 0.0024375}, 
   {"Cl", "I", "", "2Po-  3d  2D ", Null, 1100.6973, 
    294.12, 6, 10, 1., 1.6287*^8, 0., 0.049305}, 
   {"Cl", "I", "", "2Po-  3d  2D ", Null, 1107.5283, 882.36, 2, 4, 
    0.24983, 9.9856*^7, 0., 0.036726}, 
   {"Cl", "I", "", "2Po-  3d  2D ", Null, 1097.3692, 0., 4, 6, 0.56969, 
    1.5606*^8, 0., 0.042261}, 
   {"Cl", "I", "", "2Po-  3d  2D ", Null, 1096.8098, 0., 4, 4, 0.18048, 
    7.4271*^7, 0., 0.013395}, 
   {"Cl", "I", "", "2Po-  3d  2P ", Null, 1090.6302, 294.12, 6, 
    6, 1., 1.7257*^8, 0., 0.030773}, 
   {"Cl", "I", "", "2Po-  3d  2P ", Null, 1102.7554, 882.36, 2, 2, 
    0.19252, 9.6417*^7, 0., 0.017578}, 
   {"Cl", "I", "", "2Po-  3d  2P ", Null, 1095.1484, 882.36, 2, 4, 0.46933, 
    1.1999*^8, 0., 0.043149}, 
   {"Cl", "I", "", "2Po-  3d  2P ", Null, 1092.1287, 0., 4, 2, 
    0.1003, 5.17098*^7, 0., 0.0046232}, 
   {"Cl", "I", "", "2Po-  3d  2P ", Null, 1084.667, 0., 4, 4, 
    0.23785, 6.2588*^7, 0., 0.011039}, 
   {"Cl", "I", "", "2Po- 5s 2[2] ", Null, 1101.3382, 882.36, 2, 4, 
    Null, 1.3373*^7, 0., 0.00486378}, 
   {"Cl", "I", "", "2Po- 5s 2[2] ", Null, 1094.7686, 0., 4, 6, Null, 
    4.1624*^6, 0., 0.0011219}, 
   {"Cl", "I", "", "2Po- 5s 2[2] ", Null, 1090.7387, 0., 4, 4, Null, 
    1.4114*^7, 0., 0.0025173}, 
   {"Cl", "I", "", "2Po- 5s 1[1] ", Null, 1095.7971, 882.36, 2, 4, 
    Null, 8.2722*^6, 0., 0.0029783}, 
   {"Cl", "I", "", "2Po- 5s 1[1] ", Null, 1095.662, 882.36, 2, 2, Null, 
    2.253*^6, 0., 0.00040548}, {"Cl", "I", "", "2Po- 5s 1[1] ", Null, 
    1085.3035, 0., 4, 4, Null, 71776., 0., 0.000012675}, 
   {"Cl", "I", "", "2Po- 5s 1[1] ", Null, 1085.1709, 0., 4, 2, Null, 
    982010., 0., 0.000086684}, {"Cl", "I", "", "2Po- 5s 0[0] ", Null, 
    1090.2707, 882.36, 2, 2, Null, 1.4225*^7, 0., 0.0025359}, 
   {"Cl", "I", "", "2Po- 5s 0[0] ", Null, 1079.8821, 0., 4, 2, Null, 
    7.4707*^6, 0., 0.00065304}, 
   {"Cl", "I", "", "2Po- 3d   2F ", Null, 1088.0589, 0., 4, 6, 1., 
    5.9798*^7, 0., 0.01592}, {"Cl", "I", "", "2Po- 4d 2[2] ", Null, 
    1052.4631, 882.36, 2, 4, Null, 1.81*^8, 0., 0.060113}, 
   {"Cl", "I", "", "2Po- 4d 2[2] ", Null, 1043.9857, 0., 4, 6, Null, 
    7.3893*^7, 0., 0.018111}, 
   {"Cl", "I", "", "2Po- 4d 2[2] ", Null, 1042.7793, 0., 4, 4, Null, 
    1.5336*^8, 0., 0.025}, 
   {"Cl", "I", "", "2Po- 4d 2[1] ", Null, 1051.3788, 882.36, 2, 2, 
    Null, 9.541*^7, 0., 0.015811}, 
   {"Cl", "I", "", "2Po- 4d 2[1] ", Null, 1041.7148, 0., 4, 2, Null, 
    6.4212*^7, 0., 0.0052232}, {"Cl", "I", "", "2Po- 4d 2[1] ", Null, 
    1040.3476, 882.36, 2, 4, Null, 2.2279*^8, 0., 
    0.072272}, {"Cl", "I", "", "2Po- 4d 2[1] ", Null, 
    1030.8845, 0., 4, 4, Null, 1.8016*^8, 0., 0.028704}, 
   {"Cl", "I", "", "2Po- 4d 1[1] ", Null, 1038.778, 882.36, 2, 2, Null, 
    2.8845*^7, 0., 0.0046663}, 
   {"Cl", "I", "", "2Po- 4d 1[1] ", Null, 1037.5871, 882.36, 2, 4, Null, 
    2.5706*^8, 0., 0.082979}, {"Cl", "I", "", "2Po- 4d 1[1] ", Null, 
    1029.3433, 0., 4, 2, Null, 1.4723*^7, 0., 0.0011693}, 
   {"Cl", "I", "", "2Po- 4d 1[1] ", Null, 1028.1739, 0., 4, 4, Null, 
    1.4059*^8, 0., 0.022281}, 
   {"Cl", "I", "", "2Po- 4d 1[2] ", Null, 1036.5735, 882.36, 2, 4, 
    Null, 1.2328*^8, 0., 0.039716}, 
   {"Cl", "I", "", "2Po- 4d 1[2] ", Null, 1027.1786, 0., 4, 4, Null, 
    2.1817*^7, 0., 0.003451}, 
   {"Cl", "I", "", "2Po- 4d 1[2] ", Null, 1025.2821, 0., 4, 6, Null, 
    1.8806*^8, 0., 0.044457}, {"Cl", "I", "", "2Po- 4d 2[3] ", Null, 
    1031.507, 0., 4, 6, Null, 6.2958*^8, 0., 0.15064}, 
   {"Cl", "I", "", "2Po- 3d   2S ", Null, 1007.6552, 294.12, 6, 
    2, 1., 3.0642*^9, 0., 0.15548}, 
   {"Cl", "I", "", "2Po- 3d   2S ", Null, 1013.6636, 882.36, 2, 2, 
    0.32565, 9.8022*^8, 0., 0.151}, 
   {"Cl", "I", "", "2Po- 3d   2S ", Null, 1004.6676, 0., 4, 2, 
    0.67435, 2.0848*^9, 0., 0.15774}, 
   {"Cl", "I", "", "2Po- 5d 1[2] ", Null, 1006.3957, 882.36, 2, 4, Null, 
    8.2517*^7, 0., 0.025059}, 
   {"Cl", "I", "", "2Po- 5d 1[2] ", Null, 997.5375, 0., 4, 4, Null, 
    6.2262*^6, 0., 0.00092884}, 
   {"Cl", "I", "", "2Po- 5d 1[2] ", Null, 996.5402, 0., 4, 6, Null, 
    9.3111*^7, 0., 0.020794}, 
   {"Cl", "I", "", "2Po- 5d 2[3] ", Null, 1002.3464, 0., 4, 6, Null, 
    3.1186*^8, 0., 0.07046}, 
   {"Cl", "I", "", "2Po- 6d 2[3] ", Null, 987.9145, 0., 4, 6, 1., 
    1.7241*^8, 0., 0.037839}, 
   {"Cl", "I", "", "2Po- 4s'' 2S ", Null, 987.14428, 
    294.12, 6, 2, 1., 5.316*^8, 0., 0.025887}, 
   {"Cl", "I", "", "2Po- 4s'' 2S ", Null, 992.9098, 882.36, 2, 2, 
    0.37265, 1.9467*^8, 0., 0.028772}, 
   {"Cl", "I", "", "2Po- 4s'' 2S ", Null, 984.2865, 0., 4, 2, 
    0.62735, 3.3641*^8, 0., 0.024431}, 
   {"Cl", "II", "1", "3P -  3p5 3Po", Null, 1071.3179, 
    342.718, 9, 9, 1., 1.1699*^8, 0., 0.01996}, 
   {"Cl", "II", "1", "3P -  3p5 3Po", Null, 1079.0797, 696.99, 3, 
    5, 0.13889, 2.8379*^7, 1.1545*^8, 0.0082567}, 
   {"Cl", "II", "1", "3P -  3p5 3Po", Null, 1075.2294, 996.47, 1, 3, 
    0.11111, 3.8246*^7, 1.1657*^8, 
    0.019887}, {"Cl", "II", "1", "3P -  3p5 3Po", Null, 
    1071.7668, 696.99, 3, 3, 0.08333, 2.8962*^7, 
    1.1657*^8, 0.0049876}, 
   {"Cl", "II", "1", "3P -  3p5 3Po", Null, 1071.0358, 0., 5, 5, 
    0.41667, 8.7069*^7, 1.1545*^8, 0.014974}, 
   {"Cl", "II", "1", "3P -  3p5 3Po", Null, 1067.9443, 696.99, 3, 
    1, 0.11111, 1.171*^8, 1.171*^8, 
    0.0066741}, {"Cl", "II", "1", "3P -  3p5 3Po", Null, 
    1063.8311, 0., 5, 3, 0.13889, 4.9361*^7, 1.1657*^8, 0.0050251}\
, {"Cl", 
      "II", "", "3P -  4s  5So", Null, 932.978, 
    696.99, 3, 5, Null, Null, Null, Null}, 
   {"Cl", "II", "", "3P -  4s  5So", Null, 926.9588, 0., 5, 5, Null, Null, 
    Null, Null}, {"Cl", "II", "", "3P -  3d  5Do", Null, 914.8574, 
    996.47, 1, 3, Null, 5.6796*^7, 0., 0.021389}, 
   {"Cl", "II", "", "3P -  3d  5Do", Null, 912.3708, 696.99, 3, 
    5, Null, 1.1346*^8, 0., 0.023598}, 
   {"Cl", "II", "", "3P -  3d  5Do", Null, 912.3495, 696.99, 3, 
    3, Null, 626180., 0., 0.000078141}, 
   {"Cl", "II", "", "3P -  3d  5Do", Null, 912.33788, 
    696.99, 3, 1, Null, 3.9249*^7, 0., 0.0016326}, 
   {"Cl", "II", "", "3P -  3d  5Do", Null, 906.6371, 0., 5, 7, Null, 
    2.0146*^8, 0., 0.034756}, {"Cl", "II", "", "3P -  3d  5Do", Null, 
    906.6137, 0., 5, 5, Null, 955720., 0., 
    0.00011777}, {"Cl", "II", "", "3P -  3d  5Do", Null, 
    906.5927, 0., 5, 3, Null, 1.5929*^7, 0., 0.0011777}, 
   {"Cl", "III", "1", "4So-  3p4 4P ", Null, 1011.2924, 0., 4, 12, 1., 
    9.3*^7, 0., 0.042777}, {"Cl", "III", "1", "4So-  3p4 4P ", Null, 
    1015.018, 0., 4, 6, 0.5, 9.1988*^7, 
    9.1988*^7, 0.02131}, 
   {"Cl", "III", "1", "4So-  3p4 4P ", Null, 1008.759, 0., 4, 4, 
    0.33333, 9.3701*^7, 9.3701*^7, 0.014295}, 
   {"Cl", "III", "1", "4So-  3p4 4P ", Null, 1005.272, 0., 4, 2, 0.16667, 
    9.4683*^7, 9.4683*^7, 0.0071724}, 
   {"Cl", "IV", "", "3P -  3p3 5So", Null, 1436.9, 
    341.99, 5, 5, Null, 22700., 0., 7.0176*^-6}, 
   {"Cl", "IV", "", "3P -  3p3 5So", Null, 1419., 492.99, 3, 5, 
    Null, 8500., 0., 4.2765*^-6}, 
   {"Cl", "IV", "", "3P -  3p3 3Do", Null, 981.2537, 909.5, 9, 15, 1., 
    1.83*^8, 0., 0.044027}, {"Cl", "IV", "", "3P -  3p3 3Do", Null, 
    986.08, 341.99, 5, 3, 0.0055699, 
    5.0131*^6, 1.8626*^8, 0.000438478}, 
   {"Cl", "IV", "", "3P -  3p3 3Do", Null, 985.737, 
    341.99, 5, 5, 0.08333, 4.5127*^7, 
    1.8394*^8, 0.0065738}, 
   {"Cl", "IV", "", "3P -  3p3 3Do", Null, 984.943, 
    341.99, 5, 7, 0.46667, 1.8095*^8, 
    1.8095*^8, 0.036844}, 
   {"Cl", "IV", "", "3P -  3p3 3Do", Null, 977.885, 492.99, 3, 3, 
    0.08333, 7.7038*^7, 1.8626*^8, 0.011044}, 
   {"Cl", "IV", "", "3P -  3p3 3Do", Null, 977.545, 
    492.99, 3, 5, 0.25, 1.3882*^8, 
    1.8394*^8, 0.033146}, 
   {"Cl", "IV", "", "3P -  3p3 3Do", Null, 973.203, 0., 1, 3, 
    0.11111, 1.0421*^8, 1.8626*^8, 0.044391}\
, {"Cl", "V", "", 
      "2Po-  3p2 4P ", Null, 1183.3, 490.89, 4, 2, 
    Null, 146870., 393530., 0.000015415}, 
   {"Cl", "V", "", "2Po-  3p2 4P ", Null, 1175.8, 
    490.89, 4, 4, Null, 102660., 112240., 0.000021278}, 
   {"Cl", "V", "", "2Po-  3p2 4P ", Null, 1164.3, 
    490.89, 4, 6, Null, 319050., 319050., 0.000097261}, 
   {"Cl", "V", "", "2Po-  3p2 4P ", Null, 1162.8, 0., 2, 2, Null, 246660., 
    393530., 0.00005}, 
   {"Cl", "V", "", "2Po-  3p2 4P ", Null, 1162.8, 0., 2, 4, Null, 9573.6, 
    112240., 3.8812*^-6}, {"Cl", "VI", "", "1S -  3p  3Po", Null, 1013.87, 
    0., 1, 3, 1., 381000., 381000., 0.00017614}, 
   {"Ar", "I", "1", "1S -3/2[3/2]o", Null, 1066.6599, 0., 1, 3, 1., 1.3*^8, 
    1.3*^8, 0.066523}, {"Ar", "I", "2", "1S -1/2[1/2]o", Null, 
    1048.2199, 0., 1, 3, 1., 4.94*^8, 
    4.94*^8, 0.24412}, 
   {"Ar", "II", "1", "2Po-  3p6 2S ", Null, 923.8358, 
    477.193, 6, 2, 1., 2.0799*^8, 0., 
    0.0088287}, {"Ar", "II", "1", "2Po-  3p6 2S ", Null, 
    932.0537, 431.58, 2, 2, 0.33333, 6.7199*^7, 
    2.0702*^8, 0.0087507}, 
   {"Ar", "II", "1", "2Po-  3p6 2S ", Null, 919.781, 0., 4, 2, 
    0.66667, 1.3983*^8, 2.0702*^8, 
    0.0088676}, {"Ar", "V", "", "3P -  3p3 5So", Null, 
    1192.3, 29.2, 5, 5, Null, 60000., 0., 
    0.000012787}, {"Ar", "V", "", "3P -  3p3 5So", Null, 1174.7, 
    763.9, 3, 5, Null, 22900., 0., 7.8958*^-6}, 
   {"Ar", "VI", "", "2Po-  3p2 4P ", Null, 1021.2, 207.5, 4, 2, 
    Null, Null, Null, Null}, {"Ar", "VI", "", "2Po-  3p2 4P ", Null, 
    1012.67, 207.5, 4, 4, Null, Null, Null, Null}, 
   {"Ar", "VI", "", "2Po-  3p2 4P ", Null, 1000.16, 207.5, 4, 6, 
    Null, Null, Null, Null}, {"Ar", "VI", "", "2Po-  3p2 4P ", Null, 
    998.43, 0., 2, 2, Null, Null, Null, Null}, 
   {"Ar", "VI", "", "2Po-  3p2 4P ", Null, 990.3, 0., 2, 4, Null, Null, 
    Null, Null},  
   {"K", "I", "1v", "2S -  4p  2Po", 7676.232, 7678.3448, 0., 2, 6, 
    1., 3.85*^7, 0., 1.0209}, 
   {"K", "I", "1v", "2S -  4p  2Po", 7698.9741, 7701.093, 0., 2, 2, 0.33333, 
    3.8159*^7, 3.8159*^7, 0.33928}, 
   {"K", "I", "1v", "2S -  4p  2Po", 7664.9111, 7667.021, 0., 2, 4, 
    0.66667, 3.8671*^7, 3.8671*^7, 0.68159}, 
   {"K", "I", "3v", "2S -  5p  2Po", 4045.1652, 4046.3081, 0., 2, 6, 1., 
    1.2499*^6, 6.1*^6, 0.009131}, 
   {"K", "I", "3v", "2S -  5p  2Po", 4047.213, 4048.356, 0., 2, 2, 
    0.33333, 1.2381*^6, 7.3381*^6, 
    0.0030421}, {"K", "I", "3v", "2S -  5p  2Po", 4044.143, 
    4045.285, 0., 2, 4, 0.66667, 1.2409*^6, 7.3409*^6, 
    0.00608898}, {"K", "I", "4v", "2S -  6p  2Po", 3446.706, 
    3447.6936, 0., 2, 6, 1., 167999.9999999999, 0., 0.00089814}, 
   {"K", "I", "4v", "2S -  6p  2Po", 3447.375, 3448.363, 0., 2, 2, 
    0.33333, 167900., 0., 0.00029932}, 
   {"K", "I", "4v", "2S -  6p  2Po", 3446.3713, 3447.359, 0., 2, 4, 
    0.66667, 168050., 0., 0.00059882}, 
   {"K", "I", "1", "2S -  7p  2Po", 3217.309, 3218.2383, 0., 2, 6, 1., 
    45999.99999999999, 0., 0.00021428}, 
   {"K", "I", "1", "2S -  7p  2Po", 3217.62, 3218.549, 0., 2, 2, 0.33333, 
    45986., 0., 0.000071417}, {"K", "I", "1", "2S -  7p  2Po", 
    3217.154, 3218.083, 0., 2, 4, 0.66667, 
    46007., 0., 0.00014286}, 
   {"K", "I", "2", "2S -  8p  2Po", 3101.208, 3102.1079, 0., 2, 6, 
    1., 18399.99999999999, 0., 0.000079636}, 
   {"K", "I", "2", "2S -  8p  2Po", 3102.046, 3102.946, 0., 2, 2, 
    0.33333, 18385., 0., 0.000026538}, 
   {"K", "I", "2", "2S -  8p  2Po", 3100.789, 3101.689, 0., 2, 4, 
    0.66667, 18408., 0., 0.000053098}, 
   {"Ca", "I", "1v", "1S -  4p  3Po", 6572.7792, 6574.595, 0., 1, 3, 
    1., 2600., 2600., 0.000050546}, 
   {"Ca", "I", "2v", "1S -  4p  1Po", 4226.728, 4227.918, 
    0., 1, 3, 1., 2.18*^8, 2.18*^8, 1.7526}, 
   {"Ca", "I", "1", "1S -  5p  3Po", 2734.8134, 2735.623, 
    0., 1, 3, 1., 1.2048*^6, 0., 0.0040551}, 
   {"Ca", "I", "2", "1S -  5p  1Po", 2721.644, 2722.45, 
    0., 1, 3, 1., 270000., 0., 0.00090004}, 
   {"Ca", "I", "3", "1S -  4p' 3Do", 2617.542, 2618.323, 
    0., 1, 3, 1., 16000., 0., 0.000049334}, 
   {"Ca", "I", "4", "1S -  4p' 3Po", 2541.481, 2542.244, 0., 1, 3, 1., 
    17000., 0., 0.000049415}, 
   {"Ca", "I", "5", "1S -  6p  1Po", 2398.559, 2399.289, 0., 1, 3, 1., 
    1.6799*^7, 0., 0.043237}, 
   {"Ca", "I", "6", "1S -  4p' 1Po", 2275.466, 2276.169, 
    0., 1, 3, 1., 3.01*^7, 0., 0.0701388}, 
   {"Ca", "I", "7", "1S -  7p  1Po", 2200.727, 2201.414, 0., 1, 3, 
    1., 1.53*^7, 0., 0.033348}, 
   {"Ca", "I", "8", "1S -  8p  1Po", 2150.7954, 2151.472, 0., 1, 3, 1., 
    6.1*^6, 0., 0.012699}, 
   {"Ca", "II", "1v", "2S -  4p  2Po", 3945.197, 3946.3136, 0., 2, 
    6, 1., 1.3559*^8, 0., 0.94907}, 
   {"Ca", "II", "1v", "2S -  4p  2Po", 3968.4682, 3969.591, 
    0., 2, 2, 0.33333, 1.3313*^8, 1.414*^8, 0.3145}, 
   {"Ca", "II", "1v", "2S -  4p  2Po", 3933.6633, 3934.777, 0., 2, 4, 
    0.66667, 1.3679*^8, 1.4569*^8, 
    0.63457}, {"Ca", "II", "1", "2S -  5p  2Po", Null, 
    1650.5683, 0., 2, 6, 1., 269490., 0., 0.00033021}, 
   {"Ca", "II", "1", "2S -  5p  2Po", Null, 1651.9909, 0., 2, 2, 0.48479, 
    390930., 2.762*^7, 0.00015994}, 
   {"Ca", "II", "1", "2S -  5p  2Po", Null, 1649.8579, 0., 2, 4, 
    0.51521, 208540., 2.8299*^7, 0.0001702}, 
   {"Ca", "II", "2", "2S -  6p  2Po", Null, 1342.1112, 0., 2, 6, 1., 
    1.54*^6, 0., 0.0012476}, {"Ca", "II", "2", "2S -  6p  2Po", Null, 
    1342.5536, 0., 2, 2, 0.33333, 1.5385*^6, 0., 0.00041573}, 
   {"Ca", "II", "2", "2S -  6p  2Po", Null, 1341.8901, 0., 2, 4, 
    0.66667, 1.5408*^6, 0., 0.00083188}, 
   {"Sc", "I", "7v", "2D -  4p  2Do", 4022.362, 4023.4991, 101.004, 
    10, 10, 1., 1.8019*^8, 0., 0.43732}, 
   {"Sc", "I", "7v", "2D -  4p  2Do", 4047.7954, 4048.939, 
    168.34, 6, 4, 0.03484, 1.54*^7, 1.7849*^8, 
    0.025233}, {"Sc", "I", "7v", "2D -  4p  2Do", 4023.677, 
    4024.814, 168.34, 6, 6, 0.54996, 1.6599*^8, 
    1.8159*^8, 0.40071}, 
   {"Sc", "I", "7v", "2D -  4p  2Do", 4020.392, 4021.528, 
    0., 4, 4, 0.36131, 1.63*^8, 1.7849*^8, 0.39521}, 
   {"Sc", "I", "7v", "2D -  4p  2Do", 3996.598, 3997.728, 0., 4, 6, 0.05389, 
    1.6599*^7, 1.8159*^8, 0.059399}, 
   {"Sc", "I", "8v", "2D -  4p  2Fo", 3910.695, 3911.8029, 
    101.004, 10, 14, 1., 1.8041*^8, 0., 0.57941}, 
   {"Sc", "I", "8v", "2D -  4p  2Fo", 3933.3693, 3934.483, 
    168.34, 6, 6, 0.03916, 1.62*^7, 1.822*^8, 
    0.037596}, {"Sc", "I", "8v", "2D -  4p  2Fo", 
    3911.815, 3912.923, 168.34, 6, 8, 
    0.56746, 1.7999*^8, 1.7999*^8, 
    0.54784}, {"Sc", "I", "8v", "2D -  4p  2Fo", 
    3907.488, 3908.595, 0., 4, 6, 0.39338, 1.66*^8, 
    1.822*^8, 0.57029}, 
   {"Sc", "I", "9v", "2D -  4p  2Po", 3271.184, 3272.127, 101.004, 
    10, 6, 1., 3.1399*^8, 0., 0.30145}, 
   {"Sc", "I", "9v", "2D -  4p  2Po", 3273.6303, 3274.574, 
    168.34, 6, 4, 0.6, 2.8107*^8, 3.1282*^8, 
    0.30122}, {"Sc", "I", "9v", "2D -  4p  2Po", 
    3269.8994, 3270.842, 0., 4, 2, 0.33333, 3.1337*^8, 3.1337*^8, 
    0.25139}, {"Sc", "I", "9v", "2D -  4p  2Po", 3255.684, 
    3256.623, 0., 4, 4, 0.066679, 3.1751*^7, 
    3.1282*^8, 0.050483}, {"Sc", "I", "10v", "2D -  4p' 2Fo", 3018.081, 
    3018.9599, 101.004, 10, 14, 1., 8.7455*^7, 0., 0.1673}\
, {"Sc", "I", 
      "10v", "2D -  4p' 2Fo", 3030.758, 3031.64, 
    168.34, 6, 6, 0.04962, 1.*^7, 1.45*^8, 
    0.013779}, {"Sc", "I", "10v", "2D -  4p' 2Fo", 3019.351, 
    3020.23, 168.34, 6, 8, 0.56917, 
    8.7999*^7, 1.45*^8, 0.15863}, 
   {"Sc", "I", "10v", "2D -  4p' 2Fo", 3015.3674, 3016.246, 0., 4, 
    6, 0.38121, 7.8999*^7, 1.45*^8, 0.15958}, 
   {"Sc", "II", "2v", "3D - 3d4p 3Fo", 3627.2133, 3628.2479, 
    105.528, 15, 21, 1., 1.4705*^8, 0., 0.40639}, 
   {"Sc", "II", "2v", "3D - 3d4p 3Fo", 3666.534, 3667.578, 
    177.76, 7, 5, 0.00268, 1.6*^6, 1.61*^8, 0.0023047}, 
   {"Sc", "II", "2v", "3D - 3d4p 3Fo", 3651.7954, 3652.836, 
    67.72, 5, 5, 0.04957, 3.*^7, 1.61*^8, 0.060012}, 
   {"Sc", "II", "2v", "3D - 3d4p 3Fo", 3645.31, 3646.349, 
    177.76, 7, 7, 0.063049, 2.74*^7, 1.6499*^8, 
    0.054616}, {"Sc", "II", "2v", "3D - 3d4p 3Fo", 3642.784, 
    3643.822, 0., 3, 5, 0.18533, 1.1399*^8, 1.61*^8, 
    0.37489}, {"Sc", "II", "2v", "3D - 3d4p 3Fo", 3630.742, 
    3631.777, 67.72, 5, 7, 0.27281, 1.2*^8, 
    1.6499*^8, 0.3322}, 
   {"Sc", "II", "2v", "3D - 3d4p 3Fo", 3613.8293, 3614.86, 177.76, 
    7, 9, 0.42658, 1.4899*^8, 1.6499*^8, 0.37277}, 
   {"Sc", "II", "3v", "3D - 3d4p 3Do", 3575.474, 3576.495, 105.528, 
    15, 15, 1., 1.688*^8, 0., 0.32371}, 
   {"Sc", "II", "3v", "3D - 3d4p 3Do", 3590.4743, 3591.499, 177.76, 
    7, 5, 0.057999, 2.9*^7, 2.1399*^8, 0.040057}, 
   {"Sc", "II", "3v", "3D - 3d4p 3Do", 3589.633, 3590.657, 67.72, 5, 3, 
    0.055159, 4.6999*^7, 2.1399*^8, 
    0.053347}, {"Sc", "II", "3v", "3D - 3d4p 3Do", 3580.925, 
    3581.947, 0., 3, 3, 0.14649, 1.2399*^8, 
    2.1399*^8, 0.23659}, 
   {"Sc", "II", "3v", "3D - 3d4p 3Do", 3576.34, 3577.361, 67.72, 5, 
    5, 0.20947, 1.06*^8, 2.1399*^8, 0.20337}, 
   {"Sc", "II", "3v", "3D - 3d4p 3Do", 3572.526, 3573.546, 177.76, 
    7, 7, 0.38057, 1.38*^8, 2.1399*^8, 0.2642}, 
   {"Sc", "II", "3v", "3D - 3d4p 3Do", 3567.6962, 3568.715, 0., 3, 
    5, 0.068669, 3.5*^7, 2.1399*^8, 0.11138}\
, {"Sc", "II", "3v", 
      "3D - 3d4p 3Do", 3558.5324, 3559.549, 
    67.72, 5, 7, 0.08176, 3.*^7, 2.1399*^8, 0.07978}, 
   {"Sc", "II", "4v", "3D - 3d4p 3Po", 3368.145, 3369.1129, 
    105.528, 15, 9, 1., 1.1986*^8, 0., 0.12238}, 
   {"Sc", "II", "4v", "3D - 3d4p 3Po", 3372.1484, 3373.117, 177.76, 
    7, 5, 0.4605, 9.9*^7, 1.35*^8, 0.12062}, 
   {"Sc", "II", "4v", "3D - 3d4p 3Po", 3368.936, 3369.904, 67.72, 5, 3, 
    0.23099, 8.3*^7, 1.32*^8, 0.084785}, 
   {"Sc", "II", "4v", "3D - 3d4p 3Po", 3361.931, 3362.897, 
    0., 3, 1, 0.10786, 1.1799*^8, 1.3*^8, 0.066122}, 
   {"Sc", "II", "4v", "3D - 3d4p 3Po", 3361.2651, 3362.231, 0., 3, 
    3, 0.093989, 3.4999*^7, 1.32*^8, 
    0.057622}, {"Sc", "II", "4v", "3D - 3d4p 3Po", 
    3359.6774, 3360.643, 67.72, 5, 5, 0.099369, 2.16*^7, 
    1.35*^8, 0.036573}, {"Sc", "II", "4v", "3D - 3d4p 3Po", 
    3352.0483, 3353.012, 0., 3, 5, 0.00731, 
    1.6*^6, 1.35*^8, 0.0044946}, 
   {"Sc", "II", "1", "3D - 4s4p 3Po", 2555.142, 2555.9091, 105.528, 15, 9, 
    1., 2.6646*^8, 0., 0.15658}, 
   {"Sc", "II", "1", "3D - 4s4p 3Po", 2563.1894, 2563.958, 0., 3, 1, 
    0.11365, 2.7*^8, 2.7*^8, 0.088699}, 
   {"Sc", "II", "1", "3D - 4s4p 3Po", 2560.2272, 2560.995, 
    67.72, 5, 3, 0.25295, 2.01*^8, 2.7*^8, 0.11858}, 
   {"Sc", "II", "1", "3D - 4s4p 3Po", 2555.7944, 2556.561, 0., 3, 
    3, 0.086389, 6.9*^7, 2.7*^8, 0.067611}, 
   {"Sc", "II", "1", "3D - 4s4p 3Po", 2552.354, 2553.12, 
    177.76, 7, 5, 0.45926, 2.21*^8, 2.636*^8, 0.15426}, 
   {"Sc", "II", "1", "3D - 4s4p 3Po", 2545.203, 2545.967, 67.72, 5, 
    5, 0.08243, 4.*^7, 2.636*^8, 0.038871}, 
   {"Sc", "II", "1", "3D - 4s4p 3Po", 2540.822, 2541.585, 
    0., 3, 5, 0.00533, 2.6*^6, 2.636*^8, 0.0041965}, 
   {"Sc", "III", "1", "2D -  4p  2Po", Null, 1605.0941, 118.584, 
    10, 6, 1., 4.5*^8, 3.4999*^8, 0.10428}, 
   {"Sc", "III", "1", "2D -  4p  2Po", Null, 1610.194, 0., 4, 2, 
    0.33333, 4.4573*^8, 7.8573*^8, 0.086628}, 
   {"Sc", "III", "1", "2D -  4p  2Po", Null, 1603.064, 
    197.64, 6, 4, 0.6, 4.0654*^8, 7.9214*^8, 0.10442}, 
   {"Sc", "III", "1", "2D -  4p  2Po", Null, 1598.001, 0., 4, 4, 
    0.066679, 4.5604*^7, 7.9214*^8, 
    0.017459}, {"Ti", "I", "12v", "3F -   y  3Fo", 3991.6474, 
    3992.7769, 222.514, 21, 21, 1., 4.5481*^7, 0., 
    0.10879}, {"Ti", "I", "12v", "3F -   y  3Fo", 4024.572, 
    4025.709, 386.875, 9, 7, 0.04614, 6.1427*^6, 0., 
    0.011608}, {"Ti", "I", "12v", "3F -   y  3Fo", 4008.928, 
    4010.061, 170.134, 7, 5, 0.03728, 7.0286*^6, 0., 
    0.012103}, {"Ti", "I", "12v", "3F -   y  3Fo", 
    3998.6362, 3999.767, 386.873, 9, 9, 0.38575, 
    4.0722*^7, 0., 0.097669}, 
   {"Ti", "I", "12v", "3F -   y  3Fo", 3989.759, 3990.887, 
    170.134, 7, 7, 0.27755, 3.7923*^7, 0., 0.090553}, 
   {"Ti", "I", "12v", "3F -   y  3Fo", 3981.762, 3982.888, 
    0., 5, 5, 0.19519, 3.7564*^7, 0., 0.089337}, 
   {"Ti", "I", "12v", "3F -   y  3Fo", 3964.2692, 3965.391, 
    170.134, 7, 9, 0.028489, 3.0855*^6, 0., 0.0093519}, 
   {"Ti", "I", "12v", "3F -   y  3Fo", 3962.851, 3963.972, 
    0., 5, 7, 0.02961, 4.1284*^6, 0., 0.013615}, 
   {"Ti", "I", "17v", "3F -   x  3Fo", 3743.414, 3744.4785, 222.514, 21, 21, 
    1., 5.2501*^7, 0., 0.11036}, 
   {"Ti", "I", "17v", "3F -   x  3Fo", 3771.6503, 3772.722, 386.873, 9, 7, 
    0.03911, 6.0219*^6, 0., 0.0099944}, 
   {"Ti", "I", "17v", "3F -   x  3Fo", 3753.633, 3754.7, 
    170.134, 7, 5, 0.037689, 8.2418*^6, 0., 
    0.012442}, {"Ti", "I", "17v", "3F -   x  3Fo", 
    3752.8583, 3753.925, 386.873, 9, 9, 0.41406, 5.0342*^7, 0., 
    0.10635}, {"Ti", "I", "17v", "3F -   x  3Fo", 
    3741.0593, 3742.123, 170.134, 7, 7, 0.26406, 
    4.1669*^7, 0., 0.087479}, 
   {"Ti", "I", "17v", "3F -   x  3Fo", 3729.8061, 3730.867, 
    0., 5, 5, 0.1916, 4.2712*^7, 0., 0.089131}, 
   {"Ti", "I", "17v", "3F -   x  3Fo", 3722.5703, 3723.629, 
    170.134, 7, 9, 0.027079, 3.3726*^6, 0., 
    0.0090137}, {"Ti", "I", "17v", "3F -   x  3Fo", 3717.392, 
    3718.449, 0., 5, 7, 0.02642, 4.2493*^6, 
    0., 0.012332}, {"Ti", "I", "19v", "3F -   y  3Go", 3646.2851, 
    3647.3247, 222.514, 21, 27, 1., 8.0422*^7, 0., 
    0.20622}, {"Ti", "I", "19v", "3F -   y  3Go", 3687.3393, 
    3688.389, 386.875, 9, 7, 0.00117, 351050., 0., 
    0.00055687}, {"Ti", "I", "19v", "3F -   y  3Go", 3671.6704, 
    3672.716, 386.875, 9, 9, 0.019439, 
    4.5912*^6, 0., 0.0092845}, 
   {"Ti", "I", "19v", "3F -   y  3Go", 3658.095, 3659.137, 
    170.134, 7, 7, 0.01896, 5.8249*^6, 0., 0.011692}, 
   {"Ti", "I", "19v", "3F -   y  3Go", 3653.4941, 3654.535, 386.875, 
    9, 11, 0.38398, 7.5351*^7, 0., 0.1844}, 
   {"Ti", "I", "19v", "3F -   y  3Go", 3642.673, 3643.711, 
    170.134, 7, 9, 0.31991, 7.7413*^7, 0., 
    0.19811}, {"Ti", "I", "19v", "3F -   y  3Go", 
    3635.462, 3636.498, 0., 5, 7, 0.25655, 8.02937*^7, 
    0., 0.22286}, {"Ti", "I", "24v", "3F -   x  3Go", 3385.66, 3386.632, 
    386.875, 9, 7, Null, 5.2421*^6, 0., 0.0070106}, 
   {"Ti", "I", "24v", "3F -   x  3Go", 3379.2114, 3380.182, 386.875, 9, 9, 
    Null, 6.1947*^6, 0., 0.010611}, 
   {"Ti", "I", "24v", "3F -   x  3Go", 3371.452, 3372.42, 386.875, 9, 11, 
    Null, 7.1923*^7, 0., 0.14988}, 
   {"Ti", "I", "24v", "3F -   x  3Go", 3360.9892, 3361.955, 
    170.134, 7, 7, Null, Null, Null, Null}, 
   {"Ti", "I", "24v", "3F -   x  3Go", 3354.634, 3355.598, 
    170.134, 7, 9, Null, 6.8922*^7, 0., 
    0.14959}, {"Ti", "I", "24v", "3F -   x  3Go", 
    3341.873, 3342.834, 0., 5, 7, Null, 6.4687*^7, 0., 
    0.15172}, {"Ti", "I", "27v", "3F -   w  3Go", 3226.2353, 
    3227.167, 386.875, 9, 7, Null, Null, Null, Null}, 
   {"Ti", "I", "27v", "3F -   w  3Go", 3214.237, 3215.165, 386.875, 
    9, 9, Null, 6.5387*^6, 0., 0.010133}, 
   {"Ti", "I", "27v", "3F -   w  3Go", 3203.8251, 3204.751, 
    170.134, 7, 7, Null, 7.2021*^6, 0., 
    0.011089}, {"Ti", "I", "27v", "3F -   w  3Go", 3199.914, 
    3200.839, 386.875, 9, 11, Null, 9.3804*^7, 0., 
    0.1761}, {"Ti", "I", "27v", "3F -   w  3Go", 
    3191.9921, 3192.915, 170.134, 7, 9, Null, 
    8.5413*^7, 0., 0.16784}, 
   {"Ti", "I", "27v", "3F -   w  3Go", 3186.451, 3187.372, 0., 5, 
    7, Null, 7.9833*^7, 0., 0.17023}, 
   {"Ti", "I", "30v=1", "3F -   v  3Fo", 2950.1274, 2950.9905, 
    222.514, 21, 21, 1., 1.1186*^8, 0., 0.14603}, 
   {"Ti", "I", "30v=1", "3F -   v  3Fo", 2967.2214, 2968.088, 
    386.875, 9, 7, 0.03205, 1.0579*^7, 0., 0.010858}, 
   {"Ti", "I", "30v=1", "3F -   v  3Fo", 2956.7951, 
    2957.659, 170.134, 7, 5, 0.038499, 
    1.7917*^7, 0., 0.016784}, 
   {"Ti", "I", "30v=1", "3F -   v  3Fo", 2956.1323, 2956.996, 386.875, 9, 9, 
    0.37516, 9.7319*^7, 0., 0.12757}, 
   {"Ti", "I", "30v=1", "3F -   v  3Fo", 2948.2541, 2949.116, 
    170.134, 7, 7, 0.27737, 9.3253*^7, 0., 
    0.12159}, {"Ti", "I", "30v=1", "3F -   v  3Fo", 
    2941.991, 2942.851, 0., 5, 5, 0.21985, 1.0414*^8, 0., 0.13522}, 
   {"Ti", "I", "30v=1", "3F -   v  3Fo", 2937.306, 2938.165, 
    170.134, 7, 9, 0.02894, 7.6515*^6, 0., 0.012732}, 
   {"Ti", "I", "30v=1", "3F -   v  3Fo", 2933.534, 2934.392, 0., 5, 
    7, 0.028249, 9.6385*^6, 0., 0.017419}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3364.746, 3365.7131, 
    225.172, 28, 36, 1., 1.6463*^8, 0., 0.35947}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3409.821, 3410.799, 
    225.73, 8, 6, 0.0010599, 1.*^6, 
    1.7399*^8, 0.0013081}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3407.2114, 3408.189, 393.44, 10, 8, 
    0.00098, 700000., 1.7399*^8, 0.00097528}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3394.581, 3395.555, 94.1, 6, 6, 
    0.027039, 2.6*^7, 1.7399*^8, 0.044942}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3387.8452, 3388.818, 
    225.73, 8, 8, 0.0372, 2.7*^7, 
    1.7399*^8, 0.046485}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3383.7683, 3384.74, 0., 4, 
    6, 0.13591, 1.32*^8, 1.7399*^8, 
    0.34007}, {"Ti", "II", "1v", "4F -   z  4Go", 
    3380.2792, 3381.25, 393.44, 10, 10, 0.03079, 1.8*^7, 
    1.7399*^8, 0.030852}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3372.8002, 3373.769, 94.1, 6, 8, 
    0.18626, 1.3799*^8, 1.7399*^8, 0.31171}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3361.2182, 3362.184, 
    225.73, 8, 10, 0.24725, 1.4799*^8, 
    1.7399*^8, 0.31141}, 
   {"Ti", "II", "1v", "4F -   z  4Go", 3349.4082, 3350.371, 393.44, 
    10, 12, 0.33353, 1.6799*^8, 1.7399*^8, 
    0.33724}, {"Ti", "II", "2v", "4F -   z  4Fo", 
    3237.142, 3238.077, 225.172, 28, 28, 1., 
    1.5666*^8, 0., 0.24626}, {"Ti", "II", "2v", "4F -   z  4Fo", 3254.2473, 
    3255.186, 393.44, 10, 8, 0.03748, 2.0226*^7, 
    2.35*^8, 0.025704}, 
   {"Ti", "II", "2v", "4F -   z  4Fo", 3252.918, 3253.856, 
    225.73, 8, 6, 0.05415, 3.9012*^7, 2.35*^8, 
    0.046442}, {"Ti", "II", "2v", "4F -   z  4Fo", 
    3251.918, 3252.856, 94.1, 6, 4, 0.03122, 3.3772*^7, 2.35*^8, 
    0.035715}, {"Ti", "II", "2v", "4F -   z  4Fo", 
    3241.9934, 3242.929, 0., 4, 4, 0.10644, 1.162*^8, 
    2.35*^8, 0.18321}, {"Ti", "II", "2v", "4F -   z  4Fo", 
    3239.0441, 3239.979, 94.1, 6, 6, 0.13512, 
    9.8607*^7, 2.35*^8, 0.15518}, 
   {"Ti", "II", "2v", "4F -   z  4Fo", 3236.578, 3237.512, 
    225.73, 8, 8, 0.20201, 1.1082*^8, 2.35*^8, 0.17414}\
, {"Ti", "II", "2v", 
      "4F -   z  4Fo", 3234.5192, 3235.453, 393.44, 
    10, 10, 0.31413, 1.3813*^8, 2.35*^8, 0.21677}, 
   {"Ti", "II", "2v", "4F -   z  4Fo", 3229.199, 3230.131, 0., 4, 
    6, 0.03391, 2.4978*^7, 2.35*^8, 0.058606}, 
   {"Ti", "II", "2v", "4F -   z  4Fo", 3222.8422, 3223.773, 94.1, 
    6, 8, 0.04781, 2.6566*^7, 2.35*^8, 0.055189}, 
   {"Ti", "II", "2v", "4F -   z  4Fo", 3217.063, 3217.992, 
    225.73, 8, 10, 0.03774, 1.6864*^7, 2.35*^8, 
    0.032727}, {"Ti", "II", "3v", "4F -   z  2Fo", 3226.7812, 
    3227.713, 225.73, 8, 6, Null, 1.6151*^6, 0., 
    0.001892}, {"Ti", "II", "3v", "4F -   z  2Fo", 
    3214.7763, 3215.705, 393.44, 10, 8, Null, 
    3.2847*^6, 0., 0.0040738}, 
   {"Ti", "II", "3v", "4F -   z  2Fo", 3213.13, 3214.058, 94.1, 6, 
    6, Null, 605180., 0., 0.00093724}, 
   {"Ti", "II", "3v", "4F -   z  2Fo", 3203.4401, 3204.366, 0., 4, 
    6, Null, 2.063*^6, 0., 0.0047637}, 
   {"Ti", "II", "3v", "4F -   z  2Fo", 3197.532, 3198.456, 
    225.73, 8, 8, Null, 1.0744*^6, 0., 0.0016478}, 
   {"Ti", "II", "3v", "4F -   z  2Fo", 3184.125, 3185.046, 94.1, 6, 
    8, Null, 462190., 0., 0.00093724}, 
   {"Ti", "II", "4v", "4F -   z  2Do", 3157.405, 3158.319, 94.1, 6, 
    4, Null, 1.2393*^6, 0., 0.0012355}, 
   {"Ti", "II", "4v", "4F -   z  2Do", 3148.0493, 3148.961, 0., 4, 
    4, Null, 1.0611*^7, 0., 0.015774}, 
   {"Ti", "II", "4v", "4F -   z  2Do", 3143.7692, 3144.68, 
    225.73, 8, 6, Null, 6.1779*^6, 0., 0.0068693}, 
   {"Ti", "II", "4v", "4F -   z  2Do", 3130.81, 3131.717, 94.1, 6, 
    6, Null, 8.2116*^6, 0., 0.012074}, 
   {"Ti", "II", "4v", "4F -   z  2Do", 3121.61, 3122.515, 
    0., 4, 6, Null, 584770., 0., 0.0012822}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3079.366, 3080.261, 
    225.172, 28, 20, Null, Null, Null, Null}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3088.0373, 3088.934, 393.44, 
    10, 8, Null, 1.2515*^8, 2.49*^8, 0.14322}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3078.658, 3079.552, 
    225.73, 8, 6, Null, 1.0865*^8, 2.49*^8, 0.11585}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3075.2294, 3076.123, 94.1, 
    6, 4, Null, 1.1248*^8, 2.49*^8, 0.10638}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3072.9841, 3073.877, 0., 4, 
    2, Null, 1.5408*^8, 2.49*^8, 0.10913}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3072.1213, 3073.014, 
    225.73, 8, 8, Null, 1.9766*^7, 2.49*^8, 
    0.027984}, {"Ti", "II", "5v", "4F -   z  4Do", 3066.354, 
    3067.245, 0., 4, 4, Null, 3.3006*^7, 2.49*^8, 
    0.0465528}, {"Ti", "II", "5v", "4F -   z  4Do", 
    3066.228, 3067.119, 94.1, 6, 6, Null, 2.5324*^7, 
    2.49*^8, 0.035715}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3059.7443, 3060.634, 
    94.1, 6, 8, Null, Null, Null, Null}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3057.403, 3058.292, 0., 4, 
    6, Null, 2.2133*^6, 2.49*^8, 0.00465528}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3079.366, 3080.261, 
    225.172, 28, 20, Null, Null, Null, Null}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3088.0373, 3088.934, 393.44, 
    10, 8, Null, 1.2515*^8, 2.5*^8, 0.14322}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3078.658, 3079.552, 
    225.73, 8, 6, Null, 1.0865*^8, 2.5*^8, 0.11585}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3075.2294, 3076.123, 94.1, 
    6, 4, Null, 1.1248*^8, 2.5*^8, 0.10638}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3072.9841, 3073.877, 0., 4, 
    2, Null, 1.5408*^8, 2.5*^8, 0.10913}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3072.1213, 3073.014, 
    225.73, 8, 8, Null, 1.9766*^7, 2.5*^8, 
    0.027984}, {"Ti", "II", "5v", "4F -   z  4Do", 3066.354, 
    3067.245, 0., 4, 4, Null, 3.3006*^7, 2.5*^8, 
    0.0465528}, {"Ti", "II", "5v", "4F -   z  4Do", 
    3066.228, 3067.119, 94.1, 6, 6, Null, 2.5324*^7, 
    2.5*^8, 0.035715}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3059.7443, 3060.634, 
    94.1, 6, 8, Null, Null, Null, Null}, 
   {"Ti", "II", "5v", "4F -   z  4Do", 3057.403, 3058.292, 0., 4, 
    6, Null, 2.2133*^6, 2.5*^8, 0.00465528}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1904.1677, 
    225.172, 28, 28, 1., 1.3518*^8, 0., 0.073481}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1914.38, 94.1, 6, 4, 
    0.0038999, 3.6227*^6, 0., 0.0013269}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1914.008, 225.73, 8, 
    6, 0.0002, 127090., 0., 0.0000523498}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1911.618, 393.44, 10, 8, 
    0.00094998, 440810., 0., 0.0001932}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1910.938, 0., 4, 4, 
    0.13779, 1.29*^8, 0., 0.070622}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1909.198, 94.1, 6, 6, 
    0.19223, 1.2031*^8, 0., 0.065743}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1905.774, 0., 4, 6, 
    0.03134, 1.9717*^7, 0., 0.016104}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1905.508, 225.73, 8, 
    8, 0.25643, 1.2107*^8, 0., 0.065904}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1900.741, 94.1, 6, 8, 0.03989, 
    1.8976*^7, 0., 0.013704}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1897.411, 393.44, 10, 10, 
    0.30912, 1.1826*^8, 0., 0.063826}, 
   {"Ti", "II", "", "4F -      4Fo", Null, 1891.392, 225.73, 8, 
    10, 0.02817, 1.0879*^7, 0., 0.0072931}, 
   {"Ti", "II", "", "4F -      4Do", Null, 1907.3564, 
    225.172, 28, 20, 1., 3.4552*^8, 0., 0.13461}, 
   {"Ti", "II", "", "4F -      4Do", Null, 1910.609, 0., 4, 2, 
    0.10364, 3.5626*^8, 0., 0.097485}, 
   {"Ti", "II", "", "4F -      4Do", Null, 1909.664, 94.1, 6, 4, 0.17795, 
    3.0632*^8, 0., 0.11165}, 
   {"Ti", "II", "", "4F -      4Do", Null, 1908.205, 
    225.73, 8, 6, 0.27163, 3.1242*^8, 0., 
    0.12791}, {"Ti", "II", "", "4F -      4Do", Null, 
    1906.461, 393.44, 10, 8, 0.38333, 3.3159*^8, 0., 
    0.14454}, {"Ti", "II", "", "4F -      4Do", Null, 1906.239, 0., 4, 4, 
    0.02418, 4.1853*^7, 0., 0.022899}, 
   {"Ti", "II", "", "4F -      4Do", Null, 1903.424, 94.1, 6, 6, 0.02393, 
    2.7728*^7, 0., 0.015061}, 
   {"Ti", "II", "", "4F -      4Do", Null, 1900.385, 
    225.73, 8, 8, 0.014739, 1.2864*^7, 0., 
    0.0069648}, {"Ti", "II", "", "4F -      4Do", Null, 1900.021, 
    0., 4, 6, 0.00047999, 548880., 0., 0.00044559}, 
   {"Ti", "II", "", "4F -      4Do", Null, 1895.643, 94.1, 6, 8, 
    0.00014999, 121490., 0., 0.0000872677}, 
   {"Ti", "III", "", "3F -  4p  1Do", Null, 1333.097, 184.9, 7, 5, 
    Null, 73190.9, 0., 0.000013928}, 
   {"Ti", "III", "", "3F -  4p  1Do", Null, 1329.819, 0., 5, 5, Null, 
    1.0607*^7, 0., 0.0028121}, 
   {"Ti", "III", "1", "3F -  4p  3Do", Null, 1298.4967, 241.804, 
    21, 15, 1., 4.1161*^8, 0., 0.074319}, 
   {"Ti", "III", "1", "3F -  4p  3Do", Null, 1298.996, 184.9, 7, 5, 
    0.053079, 6.5457*^7, 0., 0.011828}, 
   {"Ti", "III", "1", "3F -  4p  3Do", Null, 1298.697, 0., 5, 3, 
    0.30461, 6.2662*^8, 0., 0.095067}, 
   {"Ti", "III", "1", "3F -  4p  3Do", Null, 1298.633, 420.49, 9, 
    7, 0.14182, 1.2505*^8, 0., 0.02459}, 
   {"Ti", "III", "1", "3F -  4p  3Do", Null, 1295.884, 0., 5, 5, 0.2137, 
    2.6549*^8, 0., 0.066839}, {"Ti", "III", "1", "3F -  4p  3Do", Null, 
    1294.674, 184.9, 7, 7, 0.26755, 2.3808*^8, 0., 
    0.059828}, {"Ti", "III", "1", "3F -  4p  3Do", Null, 1291.582, 0., 5, 7, 
    0.01925, 1.725*^7, 0., 0.0060399}, 
   {"Ti", "III", "2", "3F -  4p  3Fo", Null, 1288.5934, 241.804, 
    21, 21, 1., 3.9172*^8, 0., 0.097514}, 
   {"Ti", "III", "2", "3F -  4p  3Fo", Null, 1294.717, 184.9, 7, 5, 
    0.32194, 5.2218*^8, 0., 0.093735}, 
   {"Ti", "III", "2", "3F -  4p  3Fo", Null, 1293.225, 420.49, 9, 
    7, 0.40952, 4.761*^8, 0., 0.092845}, 
   {"Ti", "III", "2", "3F -  4p  3Fo", Null, 1291.625, 0., 5, 5, 0.00007, 
    121310., 0., 0.000030341}, {"Ti", "III", "2", "3F -  4p  3Fo", Null, 
    1289.299, 184.9, 7, 7, 0.00721, 8.45948*^6, 0., 
    0.0021082}, {"Ti", "III", "2", "3F -  4p  3Fo", Null, 1286.369, 
    420.49, 9, 9, 0.24264, 2.2293*^8, 0., 0.055304}, 
   {"Ti", "III", "2", "3F -  4p  3Fo", Null, 1286.233, 0., 5, 7, 
    0.00242, 2.8537*^6, 0., 0.0009909}, 
   {"Ti", "III", "2", "3F -  4p  3Fo", Null, 1282.484, 184.9, 7, 9, 
    0.01621, 1.5024*^7, 0., 0.0047632}, 
   {"Ti", "III", "", "3F -  4p  3Po", Null, 1237.018, 184.9, 7, 5, Null, 
    1.2572*^6, 0., 0.00020602}, 
   {"Ti", "III", "", "3F -  4p  3Po", Null, 1235.495, 0., 5, 3, 
    Null, 1.2143*^6, 0., 0.00016674}, 
   {"Ti", "III", "", "3F -  4p  3Po", Null, 1234.195, 0., 5, 5, Null, 
    184670., 0., 0.000042173}, {"Ti", "III", "", "3F -  4p  1Fo", Null, 
    1209.241, 420.49, 9, 7, Null, 857080., 0., 
    0.00014614}, {"Ti", "III", "", "3F -  4p  1Fo", Null, 
    1205.807, 184.9, 7, 7, Null, 283460., 0., 0.000061788}, 
   {"Ti", "III", "", "3F -  4p  1Fo", Null, 1203.124, 0., 5, 7, Null, 
    40497., 0., 0.000012304}, 
   {"Ti", "III", "", "3F -  4p  1Po", Null, 1193.362, 0., 5, 3, Null, 
    1.2175*^6, 0., 0.00015597}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3879.6933, 3880.7939, 319.341, 
    28, 28, 1., 3.4498*^7, 0., 0.077891}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3909.8603, 3910.968, 
    552.96, 10, 8, 0.0367, 4.33*^6, 0., 0.0079433}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3902.2534, 3903.359, 552.96, 
    10, 10, 0.28249, 2.6808*^7, 0., 0.061235}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3892.861, 3893.964, 
    323.46, 8, 6, 0.05162, 8.2264*^6, 0., 
    0.014025}, {"V", "I", "7v", "4F -   y  4Fo", 3875.8964, 
    3876.995, 137.38, 6, 4, 0.033969, 
    8.2242*^6, 0., 0.012355}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3875.078, 3876.176, 
    323.46, 8, 8, 0.19446, 2.3564*^7, 0., 
    0.053077}, {"V", "I", "7v", "4F -   y  4Fo", 3867.6054, 
    3868.702, 323.46, 8, 10, 0.026069, 
    2.541*^6, 0., 0.0071271}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3864.8562, 3865.952, 137.38, 
    6, 6, 0.16584, 2.7007*^7, 0., 0.060513}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3855.362, 3856.455, 0., 4, 4, 
    0.13416, 3.3015*^7, 0., 0.073611}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3847.327, 3848.418, 
    137.38, 6, 8, 0.0387, 4.7917*^6, 0., 0.014186}, 
   {"V", "I", "7v", "4F -   y  4Fo", 3844.438, 3845.528, 0., 4, 6, 
    0.036099, 5.9715*^6, 0., 0.019858}, 
   {"V", "I", "9v", "4F -   y  4Do", 3838.164, 3839.2538, 319.341, 28, 20, 
    1., 6.7076*^7, 0., 0.10587}, 
   {"V", "I", "9v", "4F -   y  4Do", 3855.841, 3856.934, 
    552.96, 10, 8, 0.34918, 5.7752*^7, 0., 0.10304}, 
   {"V", "I", "9v", "4F -   y  4Do", 3840.7514, 3841.841, 323.46, 
    8, 6, 0.24567, 5.4817*^7, 0., 0.090972}, 
   {"V", "I", "9v", "4F -   y  4Do", 3828.556, 3829.642, 
    137.38, 6, 4, 0.15775, 5.33058*^7, 0., 
    0.078136}, {"V", "I", "9v", "4F -   y  4Do", 3822.0083, 
    3823.093, 323.46, 8, 8, 0.04745, 8.0579*^6, 0., 0.017657}, 
   {"V", "I", "9v", "4F -   y  4Do", 3818.2421, 3819.326, 
    0., 4, 2, 0.09881, 6.7319*^7, 0., 0.073611}, 
   {"V", "I", "9v", "4F -   y  4Do", 3813.489, 3814.571, 
    137.38, 6, 6, 0.05191, 1.1833*^7, 0., 
    0.025814}, {"V", "I", "9v", "4F -   y  4Do", 3808.518, 
    3809.599, 0., 4, 4, 0.04312, 1.4802*^7, 0., 0.032206}, 
   {"V", "I", "9v", "4F -   y  4Do", 3795.0102, 3796.088, 137.38, 
    6, 8, 0.00271, 470310., 0., 0.0013547}, 
   {"V", "I", "9v", "4F -   y  4Do", 3793.608, 3794.685, 0., 4, 6, 
    0.00341, 790020., 0., 0.0025582}, 
   {"V", "I", "12v", "4F -   y  4Go", 3263.236, 3264.177, 0., 4, 6, Null, 
    7.0541*^6, 0., 0.016902}, 
   {"V", "I", "14v", "4F -   x  4Go", 3226.1044, 3227.036, 552.96, 
    10, 8, Null, 697340., 0., 0.00087096}, 
   {"V", "I", "14v", "4F -   x  4Go", 3217.157, 3218.086, 
    323.46, 8, 6, Null, 1.2906*^6, 0., 
    0.0015028}, {"V", "I", "14v", "4F -   x  4Go", 
    3207.4152, 3208.342, 552.96, 10, 10, Null, 2.5798*^7, 0., 
    0.039811}, {"V", "I", "14v", "4F -   x  4Go", 
    3202.3864, 3203.312, 323.46, 8, 8, Null, 
    3.9797*^7, 0., 0.061222}, 
   {"V", "I", "14v", "4F -   x  4Go", 3198.006, 3198.93, 
    137.38, 6, 6, Null, 3.8546*^7, 0., 
    0.0591368}, {"V", "I", "14v", "4F -   x  4Go", 3185.385, 
    3186.306, 552.96, 10, 12, Null, 2.6816*^8, 0., 0.48978}, 
   {"V", "I", "14v", "4F -   x  4Go", 3184.0122, 3184.933, 0., 4, 6, Null, 
    2.3977*^8, 0., 0.54694}, 
   {"V", "I", "17v", "4F -   w  4Fo", 3053.654, 3054.542, 0., 4, 4, Null, 
    1.2948*^8, 0., 0.18111}, 
   {"V", "I", "17v", "4F -   w  4Fo", 3043.5493, 3044.435, 0., 4, 
    6, Null, 1.7741*^7, 0., 0.036978}, 
   {"V", "II", "2", "5D -   z  3Do", 2747.7443, 2748.557, 
    106.63, 5, 3, Null, Null, Null, Null}, 
   {"V", "II", "2", "5D -   z  3Do", 2742.424, 2743.235, 
    36.059, 3, 3, Null, 6.2733*^6, 0., 
    0.00707758}, {"V", "II", "2", "5D -   z  3Do", 2739.7143, 
    2740.525, 0., 1, 3, Null, 3.0224*^7, 0., 0.10209}, 
   {"V", "II", "2", "5D -   z  3Do", 2714.2084, 2715.013, 
    208.89, 7, 5, Null, 8.4651*^6, 
    2.3899*^8, 0.00668198}, 
   {"V", "II", "2", "5D -   z  3Do", 2711.7392, 2712.543, 339.21, 9, 7, 
    Null, 8.3616*^6, 2.1799*^8, 0.0071739}, 
   {"V", "II", "2", "5D -   z  3Do", 2706.6933, 2707.496, 
    106.63, 5, 5, Null, 1.8708*^7, 2.3899*^8, 0.02056}, 
   {"V", "II", "2", "5D -   z  3Do", 2702.187, 2702.988, 
    208.89, 7, 7, Null, 2.8534*^7, 
    2.1799*^8, 0.031254}, 
   {"V", "II", "2", "5D -   z  3Do", 2701.531, 2702.332, 36.059, 
    3, 5, Null, 1.9575*^6, 2.3899*^8, 
    0.0035717}, {"V", "II", "2", "5D -   z  3Do", 
    2694.7382, 2695.538, 106.63, 5, 7, Null, 185250., 
    2.1799*^8, 0.00028251}, 
   {"V", "II", "1", "5D -   z  5Fo", 2710.9482, 2711.7524, 
    206.256, 25, 35, 1., 3.1266*^7, 0., 0.048256}, 
   {"V", "II", "1", "5D -   z  5Fo", 2741.576, 2742.387, 
    208.89, 7, 5, 0.00153, 322790., 2.08*^8, 0.00025996}\
, {"V", "II", "1", 
      "5D -   z  5Fo", 2733.908, 2734.717, 106.63, 5, 
    5, 0.01251, 2.669*^6, 2.08*^8, 0.0029925}, 
   {"V", "II", "1", "5D -   z  5Fo", 2732.925, 2733.734, 339.21, 9, 7, 
    0.0018799, 285450., 2.22*^8, 0.00024875}, 
   {"V", "II", "1", "5D -   z  5Fo", 2728.6411, 2729.449, 
    36.059, 3, 5, 0.089499, 1.9188*^7, 
    2.08*^8, 0.035717}, 
   {"V", "II", "1", "5D -   z  5Fo", 2723.2233, 2724.03, 
    208.89, 7, 7, 0.012269, 1.8907*^6, 2.22*^8, 
    0.0021033}, {"V", "II", "1", "5D -   z  5Fo", 2715.7521, 
    2716.557, 339.21, 9, 9, 0.01711, 2.0695*^6, 2.33*^8, 0.0022896}, 
   {"V", "II", "1", "5D -   z  5Fo", 2715.6582, 2716.463, 106.63, 5, 
    7, 0.18761, 2.9177*^7, 2.22*^8, 0.045189}, 
   {"V", "II", "1", "5D -   z  5Fo", 2713.048, 2713.852, 106.63, 5, 
    3, 0.017989, 6.5441*^6, 2.1399*^8, 
    0.0043354}, {"V", "II", "1", "5D -   z  5Fo", 
    2707.8601, 2708.663, 36.059, 3, 3, 0.03483, 
    1.2759*^7, 2.1399*^8, 0.014024}, 
   {"V", "II", "1", "5D -   z  5Fo", 2706.1713, 2706.974, 
    208.89, 7, 9, 0.26776, 3.2729*^7, 
    2.33*^8, 0.046228}, 
   {"V", "II", "1", "5D -   z  5Fo", 2705.219, 2706.021, 0., 1, 3, 0.01129, 
    4.1434*^6, 2.1399*^8, 0.013646}, 
   {"V", "II", "1", "5D -   z  5Fo", 2700.935, 2701.736, 339.21, 9, 
    11, 0.34586, 3.4791*^7, 2.33*^8, 0.046533}, 
   {"V", "II", "3", "5D -   z  5Do", 2684.222, 2685.0197, 
    206.256, 25, 25, 1., 8.3146*^7, 0., 0.089866}, 
   {"V", "II", "3", "5D -   z  5Do", 2690.7912, 2691.59, 106.63, 5, 3, 
    0.073379, 5.0467*^7, 2.9*^8, 0.032887}, 
   {"V", "II", "3", "5D -   z  5Do", 2690.2573, 2691.056, 
    208.89, 7, 5, 0.08006, 3.3062*^7, 2.9*^8, 0.025639}\
, {"V", "II", "3", 
      "5D -   z  5Do", 2689.8833, 2690.682, 
    36.059, 3, 1, 0.04543, 9.3847*^7, 2.9*^8, 0.033953}, 
   {"V", "II", "3", "5D -   z  5Do", 2688.722, 2689.52, 339.21, 9, 
    7, 0.047349, 1.3986*^7, 2.7*^8, 0.011797}, 
   {"V", "II", "3", "5D -   z  5Do", 2687.962, 2688.76, 339.21, 9, 
    9, 0.34759, 7.9946*^7, 2.9*^8, 0.086648}, 
   {"V", "II", "3", "5D -   z  5Do", 2685.6884, 2686.486, 
    36.059, 3, 3, 0.0092498, 6.3922*^6, 
    2.9*^8, 0.0069164}, {"V", "II", "3", "5D -   z  5Do", 2683.09, 2683.887, 
    0., 1, 3, 0.045639, 3.1659*^7, 2.9*^8, 0.10257}, 
   {"V", "II", "3", "5D -   z  5Do", 2682.873, 2683.67, 106.63, 5, 
    5, 0.04637, 1.9307*^7, 2.9*^8, 0.020846}, 
   {"V", "II", "3", "5D -   z  5Do", 2679.33, 2680.126, 
    208.89, 7, 7, 0.10585, 3.1603*^7, 
    2.7*^8, 0.034033}, {"V", "II", "3", "5D -   z  5Do", 2678.5751, 
    2679.371, 208.89, 7, 9, 0.049049, 
    1.1398*^7, 2.9*^8, 0.015773}, 
   {"V", "II", "3", "5D -   z  5Do", 2677.8012, 2678.597, 36.059, 
    3, 5, 0.07788, 3.2611*^7, 2.9*^8, 0.058463}, 
   {"V", "II", "3", "5D -   z  5Do", 2672.007, 2672.801, 106.63, 5, 
    7, 0.072199, 2.1733*^7, 2.7*^8, 
    0.032586}, {"V", "II", "8", "5D -   y  5Do", 
    2147.537, 2148.213, 36.059, 3, 1, Null, 1.0423*^7, 
    0., 0.0024037}, {"V", "II", "8", "5D -   y  5Do", 2142.747, 
    2143.422, 36.059, 3, 3, Null, 939310., 0., 
    0.00064696}, {"V", "II", "8", "5D -   y  5Do", 2141.0922, 
    2141.767, 0., 1, 3, Null, 3.9037*^6, 0., 0.0080538}, 
   {"V", "II", "8", "5D -   y  5Do", 2124.012, 2124.683, 36.059, 
    3, 5, Null, 1.051*^6, 0., 0.0011854}, 
   {"V", "II", "6", "5D -   z  3Po", 2140.4724, 2141.147, 36.059, 
    3, 5, Null, 7.0775*^6, 0., 0.00810738}, 
   {"V", "II", "6", "5D -   z  3Po", 2127.354, 2128.026, 
    36.059, 3, 1, Null, 6.4445*^6, 0., 
    0.0014584}, {"V", "II", "6", "5D -   z  3Po", 2123.737, 
    2124.408, 36.059, 3, 3, Null, 29211., 0., 
    0.000019764}, {"V", "II", "6", "5D -   z  3Po", 
    2122.111, 2122.782, 0., 1, 3, Null, 548540., 0., 
    0.0011117}, {"V", "II", "7", "5D -   z  5Po", 2139.33, 
    2140.0053, 206.256, 25, 15, 1., 1.8741*^8, 
    0., 0.077204}, {"V", "II", "7", "5D -   z  5Po", 2143.041, 
    2143.716, 106.63, 5, 3, 0.069379, 
    6.4668*^7, 0., 0.026732}, 
   {"V", "II", "7", "5D -   z  5Po", 2141.979, 2142.654, 
    208.89, 7, 5, 0.18321, 1.0263*^8, 0., 0.050455}, 
   {"V", "II", "7", "5D -   z  5Po", 2140.0734, 2140.748, 339.21, 
    9, 7, 0.35121, 1.4099*^8, 0., 0.075293}, 
   {"V", "II", "7", "5D -   z  5Po", 2139.8034, 2140.478, 
    36.059, 3, 3, 0.093449, 8.7499*^7, 0., 0.060101}, 
   {"V", "II", "7", "5D -   z  5Po", 2138.153, 2138.827, 0., 1, 3, 
    0.042489, 3.9872*^7, 0., 0.0820358}, 
   {"V", "II", "7", "5D -   z  5Po", 2137.296, 2137.97, 106.63, 5, 
    5, 0.10889, 6.1499*^7, 0., 0.042076}, 
   {"V", "II", "7", "5D -   z  5Po", 2134.119, 2134.792, 208.89, 
    7, 7, 0.10724, 4.3384*^7, 0., 0.029642}, 
   {"V", "II", "7", "5D -   z  5Po", 2134.075, 2134.748, 
    36.059, 3, 5, 0.027, 1.5293*^7, 0., 0.017413}, 
   {"V", "II", "7", "5D -   z  5Po", 2129.4694, 2130.142, 106.63, 5, 7, 
    0.01716, 6.9869*^6, 0., 0.0066532}, 
   {"V", "II", "", "5D -   y  3Po", Null, 1970.88, 0., 1, 3, Null, 
    191740., 0., 0.00033497}, 
   {"V", "II", "9", "5D -   z  3So", Null, 1916.4, 0., 1, 3, Null, 120520., 
    0., 0.00019907}, {"V", "II", "", "5D-4f 2[1/2]o", Null, 
    1106.571, 0., 1, 3, Null, 2.2399*^8, 0., 0.12331}, 
   {"V", "II", "", "5D-4f 2[3/2]o", Null, 1106.449, 0., 1, 3, Null, 
    4.3367*^7, 0., 0.023878}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1163.9996, 337.25, 28, 
    36, 1., 6.8438*^7, 0., 0.017873}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1173.95, 341.5, 8, 6, 0., 
    1.3204, 0., 2.046*^-10}, {"V", "III", "1", "4F -   z  4Go", Null, 
    1172.444, 583.89, 10, 8, 0.000090, 
    25696., 0., 4.2364*^-6}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1171.255, 145.5, 6, 6, 0.01034, 
    4.1657*^6, 0., 0.00085674}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1169.262, 0., 4, 6, 
    0.15438, 6.2542*^7, 0., 0.019228}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1169.122, 341.5, 8, 8, 
    0.00862, 2.6202*^6, 0., 0.000536928}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1166.554, 583.89, 10, 
    10, 0.00291, 711760., 0., 0.00014521}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1166.45, 145.5, 6, 8, 
    0.21605, 6.6118*^7, 0., 0.017982}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1163.265, 341.5, 8, 10, 0.27949, 
    6.8999*^7, 0., 0.017495}, 
   {"V", "III", "1", "4F -   z  4Go", Null, 1159.749, 583.89, 10, 
    12, 0.32813, 6.8114*^7, 0., 0.016482}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1151.1981, 337.25, 28, 28, 1., 
    6.3206*^8, 0., 0.12558}, {"V", "III", "2", "4F -   z  4Fo", Null, 
    1155.117, 145.5, 6, 4, 0.02463, 1.0785*^8, 0., 0.014383}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1154.781, 341.5, 8, 6, 0.03223, 
    9.4187*^7, 0., 0.014122}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1154.266, 583.89, 10, 
    8, 0.02518, 5.5264*^7, 0., 0.0088308}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1153.179, 0., 4, 4, 
    0.11749, 5.1676*^8, 0., 0.10302}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1152.173, 145.5, 6, 6, 
    0.15535, 4.5704*^8, 0., 0.09096}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1151.047, 341.5, 8, 8, 
    0.22649, 5.0103*^8, 0., 0.09952}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1150.245, 0., 4, 6, 
    0.02664, 7.8779*^7, 0., 0.023439}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1149.945, 
    583.89, 10, 10, 0.33224, 5.8991*^8, 0., 0.11695}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1148.456, 145.5, 6, 8, 
    0.03427, 7.6353*^7, 0., 0.020139}, 
   {"V", "III", "2", "4F -   z  4Fo", Null, 1146.75, 341.5, 8, 10, 
    0.025669, 4.5942*^7, 0., 0.011322}, 
   {"V", "III", "", "4F -   z  2Fo", Null, 1137.904, 0., 4, 6, Null, 
    1.9444*^6, 0., 0.00056616}, 
   {"V", "III", "", "4F -      2Do", Null, 1129.19, 0., 4, 4, Null, 
    5.3524*^6, 0., 0.0010232}, {"V", "III", "", "4F -      2Do", Null, 
    1117.847, 0., 4, 6, Null, 3.5013*^6, 0., 
    0.00098388}, {"V", "III", "3", "4F -      4Do", Null, 
    1125.4628, 337.25, 28, 20, 1., 4.6555*^8, 0., 
    0.063148}, {"V", "III", "3", "4F -      4Do", Null, 
    1128.631, 341.5, 8, 6, 0.17977, 2.7662*^8, 0., 
    0.03962}, {"V", "III", "3", "4F -      4Do", Null, 
    1126.14, 145.5, 6, 6, 0.02008, 3.1103*^7, 0., 
    0.00591368}, {"V", "III", "3", "4F -      4Do", Null, 
    1125.699, 583.89, 10, 8, 0.4381, 5.09588*^8, 0., 
    0.077446}, {"V", "III", "3", "4F -      4Do", Null, 1124.298, 0., 4, 6, 
    0.00038, 593240., 0., 0.00016863}, 
   {"V", "III", "3", "4F -      4Do", Null, 1123.524, 0., 4, 2, 
    0.12669, 5.9286*^8, 0., 0.056097}, 
   {"V", "III", "3", "4F -      4Do", Null, 1122.99, 145.5, 6, 4, 
    0.15294, 3.5837*^8, 0., 0.045179}, 
   {"V", "III", "3", "4F -      4Do", Null, 1122.637, 341.5, 8, 8, 
    0.0371, 4.3509*^7, 0., 0.0082207}, 
   {"V", "III", "3", "4F -      4Do", Null, 1121.158, 0., 4, 4, 
    0.043739, 1.0298*^8, 0., 0.019406}, 
   {"V", "III", "3", "4F -      4Do", Null, 1120.172, 145.5, 6, 8, 
    0.0012, 1.4206*^6, 0., 0.00035633}, 
   {"V", "III", "", "4F -   y  4Do", Null, 1009.354, 0., 4, 2, Null, 
    2.8975*^8, 0., 0.022128}, 
   {"V", "III", "", "4F -   y  4Do", Null, 1008.253, 0., 4, 4, Null, 
    5.3944*^7, 0., 0.0082213}, 
   {"V", "III", "", "4F -   y  4Do", Null, 1005.631, 0., 4, 6, 
    Null, 2.5241*^6, 0., 0.00057404}, 
   {"V", "III", "", "4F -   y  2Do", Null, 1004.939, 0., 4, 4, 
    Null, 4.8843*^6, 0., 0.00073957}, 
   {"V", "III", "", "4F -   y  2Do", Null, 1001.958, 0., 4, 6, 
    Null, 333650., 0., 0.000075325}, 
   {"Cr", "I", "1v", "7S -   z  7Po", 4269.5292, 4270.7314, 0., 7, 
    21, 1., 3.121*^7, 0., 0.25603}, 
   {"Cr", "I", "1v", "7S -   z  7Po", 4289.716, 4290.923, 0., 7, 5, 
    0.24416, 3.1555*^7, 0., 0.062216}, 
   {"Cr", "I", "1v", "7S -   z  7Po", 4274.796, 4275.999, 0., 7, 7, 
    0.32821, 3.0618*^7, 0., 0.083927}, 
   {"Cr", "I", "1v", "7S -   z  7Po", 4254.331, 4255.529, 0., 7, 9, 
    0.42763, 3.1477*^7, 0., 0.10988}, 
   {"Cr", "I", "2v", "7S -   z  5Po", 3732.023, 3733.084, 0., 7, 7, Null, 
    160290., 0., 0.00033489}, 
   {"Cr", "I", "2v", "7S -   z  5Po", 3730.8, 3731.861, 0., 7, 5, 
    Null, 158970., 0., 0.00023708}, 
   {"Cr", "I", "4v", "7S -   y  7Po", 3589.9262, 3590.9511, 0., 7, 
    21, 1., 1.519*^8, 0., 0.88098}, 
   {"Cr", "I", "4v", "7S -   y  7Po", 3605.322, 3606.35, 
    0., 7, 5, 0.25633, 1.6145*^8, 0., 0.22485}, 
   {"Cr", "I", "4v", "7S -   y  7Po", 3593.4814, 3594.507, 0., 7, 7, 
    0.32913, 1.4954*^8, 0., 0.28967}, 
   {"Cr", "I", "4v", "7S -   y  7Po", 3578.6833, 3579.705, 0., 7, 
    9, 0.41455, 1.4832*^8, 0., 0.36635}, 
   {"Cr", "I", "1", "7S -   x  7Po", 2365.619, 2366.3418, 0., 7, 21, 
    1., 5.7299*^6, 0., 0.014431}, 
   {"Cr", "I", "1", "7S -   x  7Po", 2366.811, 2367.534, 0., 7, 5, 0.28565, 
    6.8641*^6, 0., 0.00412}, {"Cr", "I", "1", "7S -   x  7Po", 
    2365.9111, 2366.634, 0., 7, 7, 0.32038, 5.5053*^6, 0., 
    0.0046228}, {"Cr", "I", "1", "7S -   x  7Po", 2364.7304, 
    2365.453, 0., 7, 9, 0.39396, 5.2732*^6, 0., 
    0.0056872}, {"Cr", "I", "2", "7S -   w  7Po", 
    2095.3122, 2095.978, 0., 7, 21, 1., 1.1396*^6, 0., 0.0022517}, 
   {"Cr", "I", "2", "7S -   w  7Po", 2095.8833, 2096.549, 
    0., 7, 5, 0.23578, 1.1276*^6, 0., 0.00053076}, 
   {"Cr", "I", "2", "7S -   w  7Po", 2095.3933, 2096.059, 0., 7, 7, 0.32539, 
    1.1123*^6, 0., 0.00073266}, 
   {"Cr", "I", "2", "7S -   w  7Po", 2094.9323, 2095.598, 
    0., 7, 9, 0.43884, 1.1676*^6, 0., 
    0.000988338}, {"Cr", "I", "", "7s -  8p  7Po", Null, 
    1940.5327, 0., 7, 21, 1., 1.7186*^7, 0., 
    0.029107}, {"Cr", "I", "", "7s -  8p  7Po", Null, 1940.64, 
    0., 7, 5, 0.2382, 1.719*^7, 0., 0.0069327}, 
   {"Cr", "I", "", "7s -  8p  7Po", Null, 1940.561, 0., 7, 7, 
    0.33336, 1.7187*^7, 0., 0.0097029}, 
   {"Cr", "I", "", "7s -  8p  7Po", Null, 1940.451, 0., 7, 9, 
    0.42844, 1.7183*^7, 0., 0.012471}, 
   {"Cr", "II", "1", "6S -   z  6Po", 2059.782, 2060.4411, 0., 6, 
    18, 1., 1.6599*^8, 0., 0.31505}, 
   {"Cr", "II", "1", "6S -   z  6Po", 2065.501, 2066.161, 0., 6, 4, 
    0.22222, 1.6363*^8, 0., 0.069817}, 
   {"Cr", "II", "1", "6S -   z  6Po", 2061.575, 2062.234, 0., 6, 6, 
    0.33333, 1.6457*^8, 0., 0.10492}, 
   {"Cr", "II", "1", "6S -   z  6Po", 2055.596, 2056.254, 0., 6, 8, 
    0.44445, 1.6601*^8, 0., 0.14031}, 
   {"Cr", "II", "", "6S -   z  4Po", 2039.914, 2040.569, 0., 6, 4, Null, 
    1.0607*^7, 0., 0.0044142}, 
   {"Cr", "II", "", "6S -   z  4Po", 2011.166, 2011.816, 0., 6, 6, Null, 
    30117., 0., 0.000018275}, 
   {"Cr", "II", "", "6S -   z  6Do", 2025.6164, 2026.269, 0., 6, 6, Null, 
    7.6489*^6, 0., 0.0047081}, 
   {"Cr", "II", "", "6S -   z  6Do", 2016.918, 2017.569, 
    0., 6, 4, Null, 5813.4, 0., 2.3651*^-6}, 
   {"Cr", "II", "", "6S -   z  6Do", 2013.62, 2014.27, 
    0., 6, 8, Null, 1.0785*^6, 0., 0.00087468}, 
   {"Cr", "II", "", "6S -   z  4Fo", Null, 1930.916, 0., 6, 8, 
    Null, 5158.5, 0., 3.8446*^-6}, 
   {"Cr", "II", "", "6S -   z  4Do", Null, 1829.2169, 0., 6, 18, 1., 
    156490., 0., 0.00023559}, 
   {"Cr", "II", "", "6S -   z  4Do", Null, 1834.879, 0., 6, 4, 0.01951, 
    13610.9, 0., 4.5798*^-6}, 
   {"Cr", "II", "", "6S -   z  4Do", Null, 1830.643, 0., 6, 6, 
    0.17709, 82944., 0., 0.000041672}, 
   {"Cr", "II", "", "6S -   z  4Do", Null, 1825.334, 0., 6, 8, 0.8034, 
    284690., 0., 0.0001896}, {"Cr", "II", "", "6S -   x  6Po", Null, 
    1081.2921, 0., 6, 18, 1., 1.5281*^7, 0., 0.0080357}, 
   {"Cr", "II", "", "6S -   x  6Po", Null, 1084.183, 0., 6, 4, 
    0.2213, 1.5097*^7, 0., 0.0017736}, 
   {"Cr", "II", "", "6S -   x  6Po", Null, 1082.041, 0., 6, 6, 
    0.33506, 1.5329*^7, 0., 0.0026906}, 
   {"Cr", "II", "", "6S -   x  6Po", Null, 1079.293, 0., 6, 8, 
    0.44363, 1.5338*^7, 0., 0.0035715}, 
   {"Cr", "II", "", "6S -  y   4So", Null, 1079.77, 0., 6, 4, 1., 
    61719., 0., 7.192*^-6}, 
   {"Cr", "II", "", "6S -   t  4Do", Null, 1069.156, 0., 6, 8, 
    Null, 29375., 0., 6.7119*^-6}, 
   {"Cr", "II", "", "6S -  5p  6Do", Null, 1068.668, 0., 6, 4, 
    Null, 86976., 0., 9.9277*^-6}, 
   {"Cr", "II", "", "6S -  5p  6Do", Null, 1064.21, 0., 6, 8, Null, 189230., 
    0., 0.000042849}, 
   {"Cr", "II", "", "6S -  5p  4Po", Null, 1066.776, 0., 6, 4, Null, 
    866840., 0., 0.000098594}, {"Cr", "II", "", "6S -  5p  4Po", Null, 
    1064.124, 0., 6, 6, Null, 3.1406*^6, 0., 0.00053315}, 
   {"Cr", "II", "", "6S -  5p  6Po", Null, 1063.4477, 0., 6, 18, 1., 
    1.7695*^7, 0., 0.0090002}, 
   {"Cr", "II", "", "6S -  5p  6Po", Null, 1063.801, 0., 6, 4, 
    0.24252, 1.9291*^7, 0., 0.002182}, 
   {"Cr", "II", "", "6S -  5p  6Po", Null, 1062.198, 0., 6, 6, 
    0.27675, 1.4743*^7, 0., 0.0024937}, 
   {"Cr", "II", "", "6S -  5p  6Po", Null, 1064.21, 0., 6, 8, 0.48074, 
    1.9098*^7, 0., 0.0043236}, 
   {"Cr", "II", "", "6S -  5p  4Do", Null, 1051.782, 0., 6, 6, 
    Null, 40658., 0., 6.7429*^-6}, 
   {"Cr", "II", "", "6S -  5p  4Do", Null, 1049.861, 0., 6, 8, Null, 
    211730., 0., 0.00004665}, {"Cr", "III", "", "5D -  4p  5Go", Null, 
    1068.57, 183.16, 5, 5, Null, 47596., 0., 8.1476*^-6}, 
   {"Cr", "III", "", "5D -  4p  5Go", Null, 1067.19, 62.22, 3, 
    5, Null, 47719., 0., 0.000013579}, 
   {"Cr", "III", "", "5D -  4p  5Go", Null, 1065.566, 183.16, 5, 
    7, Null, 139280., 0., 0.000033192}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1033.0625, 351.321, 
    25, 35, 1., 4.6324*^8, 0., 0.10376}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1042.035, 183.16, 5, 
    3, 0.00456, 2.4003*^7, 0., 0.0023444}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1041.344, 356.55, 7, 5, 
    0.025679, 8.128*^7, 0., 0.0094385}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1040.723, 62.22, 3, 
    3, 0.0073599, 3.8846*^7, 0., 0.0063078}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1040.05, 0., 1, 3, 
    0.04742, 2.5114*^8, 0., 0.12218}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1039.467, 183.16, 5, 
    5, 0.00041, 1.2899*^6, 0., 
    0.00020894}, {"Cr", "III", "2", "5D -  4p  5Fo", Null, 
    1038.162, 62.22, 3, 5, 0.12798, 4.0892*^8, 0., 
    0.11012}, {"Cr", "III", "2", "5D -  4p  5Fo", Null, 
    1035.783, 576.08, 9, 7, 0.0519, 1.1926*^8, 0., 
    0.01492}, {"Cr", "III", "2", "5D -  4p  5Fo", Null, 
    1033.433, 356.55, 7, 7, 0.21241, 4.9145*^8, 0., 
    0.078687}, {"Cr", "III", "2", "5D -  4p  5Fo", Null, 
    1033.232, 576.08, 9, 9, 0.24496, 4.4107*^8, 0., 
    0.070592}, {"Cr", "III", "2", "5D -  4p  5Fo", Null, 
    1031.584, 183.16, 5, 7, 0.00122, 
    2.8447*^6, 0., 0.00063537}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1030.893, 356.55, 7, 9, 
    0.040659, 7.37018*^7, 0., 0.015097}, 
   {"Cr", "III", "2", "5D -  4p  5Fo", Null, 1030.467, 576.08, 9, 11, 
    0.23547, 3.4969*^8, 0., 0.068039}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1035.7427, 
    351.321, 25, 25, 1., 8.18058*^8, 0., 0.13156}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1040.172, 576.08, 9, 7, 
    0.033499, 9.6346*^7, 0., 0.012155}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1037.802, 356.55, 7, 
    7, 0.041099, 1.1935*^8, 0., 0.019271}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1036.031, 576.08, 9, 
    9, 0.21679, 4.9222*^8, 0., 0.079206}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1035.938, 183.16, 5, 
    7, 0.16482, 4.8126*^8, 0., 0.1084}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1035.567, 356.55, 7, 
    5, 0.078869, 3.2271*^8, 0., 0.03706}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1035.291, 
    183.16, 5, 3, 0.08785, 5.9965*^8, 0., 
    0.057814}, {"Cr", "III", "1", "5D -  4p  5Do", Null, 
    1034.857, 62.22, 3, 1, 0.05159, 1.0577*^9, 0., 
    0.056608}, {"Cr", "III", "1", "5D -  4p  5Do", Null, 1033.996, 
    62.22, 3, 3, 0.02852, 1.9543*^8, 0., 
    0.031324}, {"Cr", "III", "1", "5D -  4p  5Do", Null, 
    1033.711, 183.16, 5, 5, 0.097969, 
    4.0306*^8, 0., 0.06457}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1033.68, 356.55, 7, 
    9, 0.17261, 3.9457*^8, 0., 0.081265}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1033.331, 0., 1, 3, 
    0.019499, 1.3321*^8, 0., 0.063973}, 
   {"Cr", "III", "1", "5D -  4p  5Do", Null, 1032.42, 62.22, 3, 
    5, 0.0071, 2.9349*^7, 0., 0.0078141}, 
   {"Cr", "III", "3", "5D -  4p  3Do", Null, 1032.047, 
    183.16, 5, 3, Null, 1.4376*^7, 0., 
    0.0013773}, {"Cr", "III", "3", "5D -  4p  3Do", Null, 
    1030.761, 62.22, 3, 3, Null, 1.7771*^7, 0., 
    0.0028306}, {"Cr", "III", "3", "5D -  4p  3Do", Null, 
    1030.1, 0., 1, 3, Null, 1.31*^8, 0., 0.0625178}, 
   {"Cr", "III", "3", "5D -  4p  3Do", Null, 1029.618, 
    183.16, 5, 5, Null, 1.9401*^6, 0., 0.00030834}, 
   {"Cr", "III", "3", "5D -  4p  3Do", Null, 1028.337, 62.22, 3, 
    5, Null, 1.2703*^8, 0., 0.033564}, 
   {"Cr", "III", "3", "5D -  4p  3Do", Null, 1025.632, 183.16, 5, 
    7, Null, 5.7818*^7, 0., 0.012765}, 
   {"Cr", "III", "", "5D -  4p  3Go", Null, 1003.427, 183.16, 5, 
    7, Null, 310510., 0., 0.000065619}, 
   {"Cr", "III", "", "5D -  4p  3Fo", Null, 987.543, 
    183.16, 5, 5, Null, 99784.8, 0., 
    0.000014589}, {"Cr", "III", "", "5D -  4p  3Fo", Null, 
    986.365, 62.22, 3, 5, Null, 20658., 0., 
    5.022*^-6}, {"Cr", "III", "", "5D -  4p  3Fo", Null, 
    984.61, 183.16, 5, 7, Null, 185570., 0., 
    0.00003776}, {"Cr", "III", "4", "5D -  4p  5Po", Null, 
    924.0093, 351.321, 25, 15, 1., 1.0413*^9, 
    0., 0.079975}, {"Cr", "III", "4", "5D -  4p  5Po", Null, 
    925.346, 183.16, 5, 3, 0.068989, 3.5761*^8, 0., 
    0.027544}, {"Cr", "III", "4", "5D -  4p  5Po", Null, 925.028, 356.55, 7, 
    5, 0.18264, 5.6867*^8, 0., 0.052108}, 
   {"Cr", "III", "4", "5D -  4p  5Po", Null, 924.311, 
    62.22, 3, 3, 0.09042, 4.7031*^8, 0., 
    0.060239}, {"Cr", "III", "4", "5D -  4p  5Po", Null, 924.046, 
    576.08, 9, 7, 0.35655, 7.9553*^8, 0., 0.079206}, 
   {"Cr", "III", "4", "5D -  4p  5Po", Null, 923.78, 0., 1, 3, 0.04055, 
    2.1129*^8, 0., 0.081096}, 
   {"Cr", "III", "4", "5D -  4p  5Po", Null, 923.547, 
    183.16, 5, 5, 0.11746, 3.6759*^8, 0., 
    0.046993}, {"Cr", "III", "4", "5D -  4p  5Po", Null, 922.5168, 
    62.22, 3, 5, 0.03079, 9.6655*^7, 0., 
    0.020553}, {"Cr", "III", "4", "5D -  4p  5Po", Null, 922.175, 
    356.55, 7, 7, 0.098, 2.2*^8, 0., 0.028048}, 
   {"Cr", "III", "4", "5D -  4p  5Po", Null, 920.703, 
    183.16, 5, 7, 0.01461, 3.2946*^7, 0., 
    0.00586188}, {"Cr", "III", "", "5D -  4p  5Do", Null, 
    920.51, 62.22, 3, 1, Null, 4.1123*^7, 0., 
    0.0017413}, {"Cr", "III", "", "5D -  4p  5Do", Null, 920.117, 
    183.16, 5, 3, Null, 2.451*^7, 0., 0.0018665}, 
   {"Cr", "III", "", "5D -  4p  5Do", Null, 919.094, 
    62.22, 3, 3, Null, 4.5539*^6, 0., 
    0.00057661}, {"Cr", "III", "", "5D -  4p  5Do", Null, 
    918.569, 0., 1, 3, Null, 1.6099*^7, 0., 0.0061094}, 
   {"Cr", "III", "", "5D -  4p  5Do", Null, 914.179, 183.16, 5, 
    5, Null, 583590., 0., 0.000073119}, 
   {"Cr", "III", "", "5D -  4p  5Do", Null, 913.178, 
    62.22, 3, 5, Null, 5.5344*^6, 0., 0.0011531}, 
   {"Cr", "III", "", "5D -  4p  5Do", Null, 912.912, 183.16, 5, 
    7, Null, 1.6263*^7, 0., 0.0028447}, 
   {"Cr", "III", "", "5D -  4p  3Po", Null, 920.672, 356.55, 7, 5, Null, 
    5.3206*^6, 0., 0.00048295}, 
   {"Cr", "III", "", "5D -  4p  3Po", Null, 919.204, 183.16, 5, 
    5, Null, 1.4268*^7, 0., 0.0018073}, 
   {"Cr", "III", "", "5D -  4p  3Po", Null, 918.184, 
    62.22, 3, 5, Null, 1.4599*^7, 0., 0.0030752}, 
   {"Cr", "III", "", "5D -  4p  3Po", Null, 914.182, 
    183.16, 5, 3, Null, 2.2282*^7, 0., 0.0016751}, 
   {"Cr", "III", "", "5D -  4p  3Po", Null, 914.111, 
    62.22, 3, 1, Null, 2.2602*^7, 0., 0.0009438}, 
   {"Cr", "III", "", "5D -  4p  3Po", Null, 913.172, 
    62.22, 3, 3, Null, 116120., 0., 0.000014517}, 
   {"Cr", "III", "", "5D -  4p  3Po", Null, 912.6538, 0., 1, 3, 
    Null, 2.1155*^6, 0., 0.00079259}, 
   {"Mn", "I", "2v", "6S -   z  6Po", 4032.351, 4033.4905, 0., 6, 18, 1., 
    1.6727*^7, 0., 0.12239}, 
   {"Mn", "I", "2v", "6S -   z  6Po", 4034.483, 4035.623, 
    0., 6, 4, 0.21054, 1.5822*^7, 0., 0.025754}, 
   {"Mn", "I", "2v", "6S -   z  6Po", 4033.0622, 4034.202, 0., 6, 6, 
    0.32823, 1.6462*^7, 0., 0.040165}, 
   {"Mn", "I", "2v", "6S -   z  6Po", 4030.753, 4031.892, 0., 6, 8, 
    0.46124, 1.7379*^7, 0., 0.056474}, 
   {"Mn", "I", "1", "6S -   y  6Po", 2797.3574, 2798.1822, 0., 6, 
    18, 1., 3.6159*^8, 0., 1.2739}, 
   {"Mn", "I", "1", "6S -   y  6Po", 2801.082, 2801.907, 0., 6, 4, 
    0.22782, 3.6913*^8, 0., 0.28963}, 
   {"Mn", "I", "1", "6S -   y  6Po", 2798.269, 2799.094, 0., 6, 6, 
    0.32897, 3.5642*^8, 0., 0.41865}, 
   {"Mn", "I", "1", "6S -   y  6Po", 2794.817, 2795.641, 
    0., 6, 8, 0.44322, 3.6149*^8, 0., 0.56474}, 
   {"Mn", "I", "", "6S -   x  6Po", 2216.264, 2216.9547, 0., 6, 18, 
    1., 7.5717*^7, 0., 0.16737}, 
   {"Mn", "I", "", "6S -   x  6Po", 2221.832, 2222.523, 0., 6, 8, 
    0.47017, 7.9499*^7, 0., 0.078496}, 
   {"Mn", "I", "", "6S -   x  6Po", 2213.85, 2214.54, 0., 6, 6, 
    0.32486, 7.4033*^7, 0., 0.054431}, 
   {"Mn", "I", "", "6S -   x  6Po", 2208.8083, 2209.497, 0., 6, 4, 
    0.20498, 7.0559*^7, 0., 0.034423}, 
   {"Mn", "I", "", "6S -   v  6Po", 2000.2971, 2000.9446, 0., 6, 18, 
    1., 7.6211*^7, 0., 0.13723}, 
   {"Mn", "I", "", "6S -   v  6Po", 2003.8413, 2004.49, 0., 6, 8, 
    0.44376, 7.5691*^7, 0., 0.060792}, 
   {"Mn", "I", "", "6S -   v  6Po", Null, 1999.5, 0., 6, 6, 
    0.33271, 7.6234*^7, 0., 0.045693}, 
   {"Mn", "I", "", "6S -   v  6Po", Null, 1996.047, 0., 6, 4, 
    0.22352, 7.7222*^7, 0., 0.03075}, 
   {"Mn", "I", "", "7S -      4Do", Null, 1492.954, 0., 6, 8, Null, 
    1.3334*^8, 0., 0.059409}, 
   {"Mn", "II", "1", "7S -   z  7Po", 2588.964, 2589.7391, 0., 7, 
    21, 1., 2.7*^8, 0., 0.81443}, 
   {"Mn", "II", "1", "7S -   z  7Po", 2605.684, 2606.462, 0., 7, 5, 0.2381, 
    2.6484*^8, 0., 0.19267}, {"Mn", "II", "1", "7S -   z  7Po", 2593.7233, 
    2594.499, 0., 7, 7, 0.33333, 2.6851*^8, 0., 
    0.27098}, {"Mn", "II", "1", "7S -   z  7Po", 2576.1054, 
    2576.877, 0., 7, 9, 0.42857, 2.7406*^8, 0., 
    0.35078}, {"Mn", "II", "2", "7S -   z  5Po", 2305.005, 
    2305.714, 0., 7, 7, Null, 1.5539*^6, 0., 0.0012385}, 
   {"Mn", "II", "2", "7S -   z  5Po", 2298.955, 2299.663, 0., 7, 5, 
    Null, 963480., 0., 0.00054563}, 
   {"Mn", "II", "", "7S -   x  5Fo", Null, 1219.604, 0., 7, 9, 
    Null, 512230., 0., 0.00014686}, 
   {"Mn", "II", "", "7S -   x  5Fo", Null, 1218.745, 0., 7, 7, 
    Null, 145620., 0., 0.000032427}, 
   {"Mn", "II", "", "7S -   x  5Fo", Null, 1218.704, 0., 7, 5, Null, 
    62716., 0., 9.9747*^-6}, 
   {"Mn", "II", "", "7S -   w  3Do", Null, 1213.305, 0., 7, 7, Null, 214.34, 
    0., 4.7304*^-8}, {"Mn", "II", "", "7S -   w  3Do", Null, 
    1203.79, 0., 7, 5, Null, 993280., 0., 0.00015414}, 
   {"Mn", "II", "", "7S -   x  5Do", Null, 1210.576, 0., 7, 9, Null, 
    3.3336*^6, 0., 0.00094168}, 
   {"Mn", "II", "", "7S -   x  5Do", Null, 1209.006, 0., 7, 7, Null, 
    2.5246*^6, 0., 0.00055323}, 
   {"Mn", "II", "", "7S -   x  5Do", Null, 1208.674, 0., 7, 5, Null, 
    915280., 0., 0.00014319}, 
   {"Mn", "II", "", "7S -   w  3Fo", Null, 1207.281, 0., 7, 9, Null, 3313.5, 
    0., 9.3099*^-7}, 
   {"Mn", "II", "", "7S -   w  3Fo", Null, 1206.014, 0., 7, 5, 
    Null, 520560., 0., 0.000081078}, 
   {"Mn", "II", "", "7S -   w  3Fo", Null, 1205.749, 0., 7, 7, 
    Null, 262740., 0., 0.000057267}, 
   {"Mn", "II", "3", "7S -   y  7Po", Null, 1198.8542, 0., 7, 21, 
    1., 5.4241*^8, 0., 0.35062}, 
   {"Mn", "II", "3", "7S -   y  7Po", Null, 1201.118, 0., 7, 5, 
    0.25179, 5.7017*^8, 0., 0.088085}, 
   {"Mn", "II", "3", "7S -   y  7Po", Null, 1199.391, 0., 7, 7, 0.30217, 
    4.9105*^8, 0., 0.10599}, 
   {"Mn", "II", "3", "7S -   y  7Po", Null, 1197.184, 0., 7, 9, 0.44613, 
    5.6699*^8, 0., 0.15664}, 
   {"Mn", "II", "4", "7S -   x  7Po", Null, 1162.9735, 0., 7, 21, 1., 
    3.6777*^7, 0., 0.022371}, 
   {"Mn", "II", "4", "7S -   x  7Po", Null, 1164.208, 0., 7, 5, 
    0.21022, 3.2368*^7, 0., 0.00469798}, 
   {"Mn", "II", "4", "7S -   x  7Po", Null, 1163.326, 0., 7, 7, 0.336, 
    3.7038*^7, 0., 0.0075145}, 
   {"Mn", "II", "4", "7S -   x  7Po", Null, 1162.015, 0., 7, 9, 
    0.45378, 3.9037*^7, 0., 0.01016}, 
   {"Mn", "II", "", "7S -   w  5Po", Null, 1150.779, 0., 7, 7, 
    Null, 126490., 0., 0.000025113}, 
   {"Mn", "II", "", "7S -   w  5Po", Null, 1150.261, 0., 7, 5, 
    Null, 62457., 0., 8.8492*^-6}, 
   {"Mn", "II", "", "7S -   v  5Po", Null, 1122.597, 0., 7, 5, Null, 
    373010., 0., 0.000050339}, {"Mn", "II", "", "7S -   v  5Po", Null, 
    1118.206, 0., 7, 7, Null, 488650., 0., 
    0.0000916018}, {"Mn", "II", "", "7S -   w  5Do", Null, 
    1053.891, 0., 7, 5, Null, 5267.2, 0., 6.2647*^-7}, 
   {"Mn", "II", "", "7S -   w  5Do", Null, 1052.746, 0., 7, 7, Null, 
    120060., 0., 0.000019948}, 
   {"Mn", "II", "", "7S -   w  5Do", Null, 1050.351, 0., 7, 9, Null, 
    356640., 0., 0.000075841}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 
    5254.9541, 5256.417, 888.129, 3, 5, Null, 831.37, 0., 
    5.7396*^-6}, {"Fe", "I", "1v", "5D -   z  7Do", 5250.2084, 
    5251.67, 978.0728, 1, 3, Null, 929.88, 0., 
    0.000011535}, {"Fe", "I", "1v", "5D -   z  7Do", 5247.0483, 5248.509, 
    704.004, 5, 7, Null, 391.72, 0., 2.2648*^-6}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5225.5253, 5226.98, 
    888.129, 3, 3, Null, 1322.9, 0., 
    5.4185*^-6}, {"Fe", "I", "1v", "5D -   z  7Do", 
    5221.4311, 5222.885, 415.932, 7, 9, Null, Null, Null, 
    Null}, {"Fe", "I", "1v", "5D -   z  7Do", 5204.582, 5206.031, 704.004, 
    5, 5, Null, 2291.7, 0., 9.3117*^-6}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5175.7124, 5177.154, 704.004, 
    5, 3, Null, Null, Null, Null}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5168.8964, 5170.336, 
    415.932, 7, 7, Null, 3828.3, 0., 0.000015343}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5166.281, 5167.72, 0., 9, 11, 
    Null, 1449.3, 0., 7.0918*^-6}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5127.6801, 5129.109, 415.932, 
    7, 5, Null, 38.027, 0., 1.0713*^-7}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5110.413, 5111.837, 0., 9, 9, 
    Null, 4928.9, 0., 0.000019309}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5060.0781, 5061.489, 0., 9, 
    7, Null, 128.97, 0., 3.8526*^-7}, 
   {"Fe", "I", "1v", "5D -   z  7Do", 5110.413, 5111.8368, 
    0., 9, 9, Null, 4928.9, 0., 0.000019309}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4489.7392, 4490.999, 
    978.0728, 1, 3, Null, 11922., 0., 0.00010814}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4482.1694, 4483.427, 
    888.129, 3, 5, Null, 20939., 0., 0.00010517}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4471.676, 4472.931, 
    888.129, 3, 3, Null, 112.42, 0., 3.3719*^-7}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4466.5732, 4467.8271, 
    888.129, 3, 1, Null, Null, Null, Null}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4461.6523, 4462.9044, 
    704.004, 5, 7, Null, 29499., 0., 0.00012332}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4445.471, 4446.7189, 
    704.004, 5, 5, Null, 244.49, 0., 7.2449*^-7}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4435.148, 4436.3934, 704.004, 
    5, 3, Null, 4720.2, 0., 8.3566*^-6}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4427.309, 4428.552, 415.932, 
    7, 9, Null, Null, Null, Null}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4405.019, 4406.256, 415.932, 7, 7, 
    Null, Null, Null, Null}, {"Fe", "I", "2v", "5D -   z  7Fo", 
    4389.245, 4390.478, 415.932, 7, 5, Null, 1807.8, 0., 
    3.7317*^-6}, {"Fe", "I", "2v", "5D -   z  7Fo", 4375.9291, 
    4377.159, 0., 9, 11, Null, 29469., 0., 0.00010346}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4347.233, 4348.455, 0., 9, 9, 
    Null, 123.09, 0., 3.4895*^-7}, 
   {"Fe", "I", "2v", "5D -   z  7Fo", 4325.7392, 4326.956, 0., 9, 7, 
    Null, Null, Null, Null}, {"Fe", "I", "2v", "5D -   z  7Fo", 4375.93, 
    4377.1591, 0., 9, 11, Null, 29469., 0., 0.00010346}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4291.4624, 4292.67, 415.932, 
    7, 9, Null, Null, Null, Null}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4258.3144, 4259.513, 704.004, 5, 7, 
    Null, 2537.9, 0., 9.6612*^-6}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4232.725, 4233.917, 
    888.129, 3, 5, Null, 878.39, 0., 3.9344*^-6}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4216.184, 4217.371, 0., 9, 9, 
    Null, 18358., 0., 0.000048951}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4206.696, 4207.881, 415.932, 
    7, 7, Null, 7094.4, 0., 0.000018832}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4199.9824, 4201.166, 704.004, 
    5, 5, Null, Null, Null, Null}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4149.76, 4150.93, 415.932, 
    7, 5, Null, Null, Null, Null}, 
   {"Fe", "I", "3v", "5D -   z  7Po", 4134.336, 4135.502, 0., 9, 7, Null, 
    Null, Null, Null}, {"Fe", "I", "3v", "5D -   z  7Po", 4216.184, 
    4217.371, 0., 9, 9, Null, 18358., 0., 
    0.000048951}, {"Fe", "I", "4v", "5D -   z  5Do", 3882.73, 
    3883.8318, 402.96, 25, 25, 1., 1.0015*^7, 0., 0.022647}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3930.2961, 3931.409, 704.004, 
    5, 7, 0.04596, 1.5847*^6, 0., 0.0051408}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3927.92, 3929.032, 888.129, 
    3, 5, 0.045939, 2.2213*^6, 0., 0.0085689}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3922.9111, 3924.022, 415.932, 
    7, 9, 0.03986, 1.0751*^6, 0., 0.0031908}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3920.258, 3921.368, 
    978.0728, 1, 3, 0.03201, 2.5951*^6, 0., 
    0.017947}, {"Fe", "I", "4v", "5D -   z  5Do", 3906.4794, 
    3907.586, 888.129, 3, 3, 0.01016, 832160., 0., 
    0.0019049}, {"Fe", "I", "4v", "5D -   z  5Do", 
    3899.707, 3900.812, 704.004, 5, 5, 
    0.052239, 2.5814*^6, 0., 0.0058888}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3895.6562, 3896.76, 
    888.129, 3, 1, 0.03789, 9.3915*^6, 0., 0.0071265}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3886.282, 3887.383, 
    415.932, 7, 7, 0.14849, 5.2934*^6, 0., 0.011992}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3878.573, 3879.672, 704.004, 5, 3, 
    0.07881, 6.5983*^6, 0., 0.0089337}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3859.912, 3861.006, 
    0., 9, 9, 0.34237, 9.6939*^6, 0., 0.021665}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3856.3713, 3857.465, 415.932, 
    7, 5, 0.0908, 4.6405*^6, 0., 0.0073944}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3824.444, 3825.529, 0., 9, 7, 
    0.07559, 2.8292*^6, 0., 0.0048279}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3859.9113, 3861.0058, 0., 9, 
    9, Null, 9.6939*^6, 1.279*^7, 0.021665}, 
   {"Fe", "I", "4v", "5D -   z  5Do", 3824.444, 3825.5288, 
    0., 9, 7, Null, 2.8292*^6, 1.233*^7, 0.0048279}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3727.985, 3729.0472, 402.96, 
    25, 35, 1., 1.5352*^7, 0., 0.044807}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3748.262, 3749.3272, 
    888.129, 3, 5, 0.08651, 9.1467*^6, 
    1.473*^7, 0.032128}, {"Fe", "I", "5v", "5D -   z  5Fo", 3745.899, 
    3746.9639, 978.0728, 1, 3, 0.04148, 
    7.3226*^6, 1.453*^7, 0.046238}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3745.561, 3746.6256, 704.004, 
    5, 7, 0.15197, 1.1502*^7, 1.508*^7, 0.033887}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3737.1311, 3738.1938, 415.932, 7, 9, 
    0.23866, 1.4144*^7, 1.572*^7, 0.038098}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3733.317, 3734.3786, 
    888.129, 3, 3, 0.034789, 6.2028*^6, 
    1.453*^7, 0.012968}, {"Fe", "I", "5v", "5D -   z  5Fo", 
    3722.5624, 3723.6214, 704.004, 5, 5, 0.04603, 4.9687*^6, 
    1.473*^7, 0.010328}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3719.935, 3720.9927, 0., 9, 
    11, 0.3302, 1.6234*^7, 1.653*^7, 0.041187}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3707.822, 3708.8766, 704.004, 
    5, 3, 0.00397, 721999.9999999999, 1.453*^7, 0.00089337}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3705.566, 3706.6201, 415.932, 
    7, 7, 0.041129, 3.2143*^6, 1.508*^7, 0.0066207}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3683.0544, 3684.103, 415.932, 
    7, 5, 0.00254, 283470., 1.473*^7, 0.000412}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3679.913, 3680.961, 0., 9, 9, 
    0.02219, 1.3771*^6, 1.5729*^7, 0.0027974}, 
   {"Fe", "I", "5v", "5D -   z  5Fo", 3649.303, 3650.3427, 0., 9, 
    7, 0.00055, 45017.8, 1.508*^7, 0.000069945}, 
   {"Fe", "I", "6v", "5D -   z  5Po", 3440.606, 3441.5919, 0., 9, 7, 
    Null, 1.71*^7, 0., 0.023617}, 
   {"Fe", "I", "7v", "5D -   z  3Fo", 3193.226, 3194.149, 0., 9, 9, 
    Null, 442999.9999999999, 0., 0.00067769}, 
   {"Fe", "I", "8v", "5D -   z  3Do", 3191.6591, 3192.5819, 
    0., 9, 7, Null, 237000., 3.9499*^6, 0.00028167}, 
   {"Fe", "I", "9v", "5D -   y  5Do", 3059.086, 3059.9751, 415.932, 
    7, 9, Null, 1.7317*^7, 1.6999*^8, 0.031254}, 
   {"Fe", "I", "9v", "5D -   y  5Do", 3021.073, 3021.9526, 415.932, 7, 7, 
    Null, 4.5548*^7, 1.75*^8, 0.062359}, 
   {"Fe", "I", "9v", "5D -   y  5Do", 3020.6391, 3021.5189, 
    0., 9, 9, Null, 7.5936*^7, 1.6999*^8, 0.10393}, 
   {"Fe", "I", "9v", "5D -   y  5Do", 2994.427, 2995.3002, 415.932, 
    7, 5, Null, 4.3883*^7, 1.54*^8, 0.04216}, 
   {"Fe", "I", "9v", "5D -   y  5Do", 2983.57, 2984.4403, 0., 9, 7, Null, 
    2.7946*^7, 1.75*^8, 0.029024}, 
   {"Fe", "I", "10v=1", "5D -   y  5Fo", 2966.8981, 2967.7647, 0., 
    9, 11, Null, 2.7158*^7, 1.27*^8, 0.043829}, 
   {"Fe", "I", "10v=1", "5D -   y  5Fo", 2936.9033, 2937.7624, 0., 
    9, 9, Null, 1.2998*^7, 1.3*^8, 0.016817}, 
   {"Fe", "I", "10v=1", "5D -   y  5Fo", 2912.1574, 
    2913.0103, 0., 9, 7, Null, 3.3141*^6, 1.3*^8, 
    0.0032791}, {"Fe", "I", "2", "5D -   z  5Go", 2874.1723, 
    2875.0159, 0., 9, 11, Null, 931000., 8.62*^7, 
    0.0014101}, {"Fe", "I", "2", "5D -   z  5Go", 2835.457, 
    2836.2905, 0., 9, 9, Null, 324000., 9.2699*^7, 0.00039075}, 
   {"Fe", "I", "2", "5D -   z  5Go", 2807.2451, 2808.0723, 0., 9, 7, Null, 
    166999.9999999999, 9.8*^7, 0.00015355}, 
   {"Fe", "I", "5", "5D -   y  5Po", 2729.737, 2730.5457, 402.96, 
    25, 15, 1., 1.8421*^8, 0., 0.12354}, 
   {"Fe", "I", "5", "5D -   y  5Po", 2772.11, 2772.9284, 704.004, 5, 
    7, 0.010949, 4.1226*^6, 1.75*^8, 0.0066532}, 
   {"Fe", "I", "5", "5D -   y  5Po", 2756.3283, 2757.143, 888.129, 
    3, 5, 0.037549, 2.0149*^7, 1.6799*^8, 
    0.038272}, {"Fe", "I", "5", "5D -   y  5Po", 2750.141, 
    2750.9537, 415.932, 7, 7, 0.1008, 3.8912*^7, 1.75*^8, 
    0.044147}, {"Fe", "I", "5", "5D -   y  5Po", 2744.068, 
    2744.8796, 978.0728, 1, 3, 0.039139, 
    3.5479*^7, 1.61*^8, 0.12023}, 
   {"Fe", "I", "5", "5D -   y  5Po", 2742.406, 2743.2168, 
    704.004, 5, 5, 0.11541, 6.29*^7, 1.6799*^8, 
    0.070963}, {"Fe", "I", "5", "5D -   y  5Po", 
    2737.31, 2738.1197, 888.129, 3, 3, 
    0.09151, 8.3583*^7, 1.61*^8, 0.093946}, 
   {"Fe", "I", "5", "5D -   y  5Po", 2723.578, 2724.3845, 704.004, 
    5, 3, 0.06907, 6.4045*^7, 1.61*^8, 0.042759}, 
   {"Fe", "I", "5", "5D -   y  5Po", 2720.903, 2721.7087, 415.932, 
    7, 5, 0.19004, 1.0604*^8, 1.6799*^8, 0.084121}, 
   {"Fe", "I", "5", "5D -   y  5Po", 2719.0273, 2719.8331, 0., 9, 7, 
    0.34557, 1.3803*^8, 1.75*^8, 0.11906}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2526.5114, 2527.2729, 
    402.96, 25, 25, 1., 3.2466*^8, 0., 0.31088}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2549.6132, 2550.3784, 415.932, 
    7, 9, 0.041069, 3.6032*^7, 3.8999*^8, 
    0.045175}, {"Fe", "I", "7", "5D -   x  5Do", 2545.979, 
    2546.7429, 704.004, 5, 7, 0.059279, 6.7154*^7, 
    3.8999*^8, 0.091418}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2540.972, 2541.7351, 
    888.129, 3, 5, 0.057899, 9.2239*^7, 
    3.8999*^8, 0.14889}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2535.6074, 2536.3694, 
    978.0728, 1, 3, 0.03639, 9.7409*^7, 
    3.8999*^8, 0.28184}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2529.836, 2530.5964, 
    888.129, 3, 3, 0.01426, 3.8422*^7, 3.8999*^8, 
    0.036887}, {"Fe", "I", "7", "5D -   x  5Do", 2529.135, 
    2529.8953, 704.004, 5, 5, 0.06024, 9.74928*^7, 
    3.8999*^8, 0.093547}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2527.435, 2528.1949, 
    415.932, 7, 7, 0.16204, 1.8768*^8, 3.8999*^8, 
    0.17985}, {"Fe", "I", "7", "5D -   x  5Do", 2524.293, 2525.052, 
    888.129, 3, 1, 0.0416, 3.3853*^8, 
    3.8999*^8, 0.10786}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2522.85, 2523.6084, 0., 9, 9, 
    0.32273, 2.9232*^8, 3.8999*^8, 0.2791}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2518.102, 2518.8598, 704.004, 
    5, 3, 0.068879, 1.882*^8, 3.8999*^8, 0.10741}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2510.835, 2511.591, 415.932, 7, 5, 
    0.07884, 1.304*^8, 3.8999*^8, 0.088085}, 
   {"Fe", "I", "7", "5D -   x  5Do", 2501.1323, 2501.8862, 0., 9, 7, 
    0.056999, 6.8999*^7, 3.8999*^8, 
    0.049632}, {"Fe", "I", "8", "5D -   y  7Po", 2486.3732, 
    2487.1238, 0., 9, 7, Null, 3.08*^6, 6.8599*^6, 
    0.0022216}, {"Fe", "I", "8", "5D -   y  7Po", 2473.1621, 2473.9095, 0., 
    9, 9, Null, 2.75*^6, 3.2499*^6, 0.0025232}, 
   {"Fe", "I", "9", "5D -   x  5Fo", 2488.143, 2488.8935, 415.932, 
    7, 9, Null, 4.6547*^8, 5.*^8, 0.55578}, 
   {"Fe", "I", "9", "5D -   x  5Fo", 2483.2714, 2484.0211, 0., 9, 11, Null, 
    4.9254*^8, 5.*^8, 0.55687}, 
   {"Fe", "I", "9", "5D -   x  5Fo", 2472.895, 2473.6422, 415.932, 
    7, 7, Null, 1.2983*^8, 5.*^8, 0.1191}, 
   {"Fe", "I", "9", "5D -   x  5Fo", 2462.6472, 2463.3922, 0., 9, 9, 
    Null, 5.8457*^7, 5.*^8, 0.053181}, 
   {"Fe", "I", "9", "5D -   x  5Fo", 2462.181, 2462.9257, 415.932, 
    7, 5, Null, 1.5215*^7, 5.*^8, 0.00988338}, 
   {"Fe", "I", "9", "5D -   x  5Fo", 2447.7094, 2448.4509, 
    0., 9, 7, Null, 3.2981*^6, 5.*^8, 0.0023055}, 
   {"Fe", "I", "14", "5D -   w  5Do", 2298.1694, 2298.8772, 0., 9, 
    9, Null, 3.0894*^7, 1.35*^8, 0.024477}, 
   {"Fe", "I", "14", "5D -   w  5Do", 2276.026, 2276.7289, 
    0., 9, 7, Null, 1.6766*^7, 1.18*^8, 0.010133}, 
   {"Fe", "I", "", "5D -      5Do", 2263.4743, 2264.1747, 0., 9, 7, 
    Null, 796570., 1.4649*^7, 0.00047617}, 
   {"Fe", "I", "", "5D -      5Fo", 2259.511, 2260.2102, 
    0., 9, 11, Null, 6.9896*^6, 2.56*^7, 0.0065427}, 
   {"Fe", "I", "", "5D -      5Fo", 2250.791, 2251.4881, 0., 9, 9, 
    Null, 1.8835*^6, 2.43*^7, 0.0014314}, 
   {"Fe", "I", "20", "5D -   x  5Go", 2186.25, 2186.934, 0., 9, 11, 
    Null, 1.0866*^6, 3.56*^7, 0.00095226}, 
   {"Fe", "I", "21", "5D -   w  5Po", 2166.7731, 2167.4531, 0., 9, 
    7, Null, 2.7362*^8, 0., 0.14988}, 
   {"Fe", "I", "24", "5D -      5Do", 2139.6982, 2140.3727, 
    0., 9, 9, Null, 2.2076*^7, 0., 0.015162}, 
   {"Fe", "I", "24", "5D -      5Do", 2138.593, 2139.2668, 
    0., 9, 7, Null, 2.8088*^6, 0., 0.0014988}, 
   {"Fe", "I", "25", "5D -      3Fo", 2132.017, 2132.6899, 
    0., 9, 9, Null, 7.6216*^6, 0., 0.0051971}, 
   {"Fe", "I", "26", "5D -   z  3Ho", 2126.6091, 2127.2809, 0., 9, 11, Null, 
    644320., 0., 0.000534278}, 
   {"Fe", "I", "28", "5D -   w  5Go", 2108.1364, 2108.8047, 0., 9, 
    11, Null, 2.6042*^6, 0., 0.0021221}, 
   {"Fe", "I", "31", "5D -   v  5Fo", 2099.905, 2100.5714, 0., 9, 
    11, Null, 1.2534*^6, 0., 0.0010133}, 
   {"Fe", "I", "31", "5D -   v  5Fo", 2085.7121, 2086.376, 0., 9, 9, 
    Null, 1.6486*^6, 0., 0.0010759}, 
   {"Fe", "I", "31", "5D -   v  5Fo", 2077.3491, 2078.0115, 0., 9, 7, Null, 
    3.7047*^6, 0., 0.0018653}, 
   {"Fe", "I", "33", "5D -   v  5Po", 2084.121, 2084.784, 0., 9, 7, 
    Null, 3.7233*^7, 0., 0.018869}, 
   {"Fe", "I", "35", "5D -   u  5Fo", Null, 1960.144, 0., 9, 11, Null, 
    4.0288*^6, 0., 0.0028363}, 
   {"Fe", "I", "35", "5D -   u  5Fo", Null, 1937.2682, 0., 9, 7, Null, 
    2.1611*^7, 0., 0.0094571}, 
   {"Fe", "I", "36", "5D -   t  5Do", Null, 1946.9878, 0., 9, 7, Null, 
    1.1949*^7, 0., 0.0052815}, 
   {"Fe", "I", "", "5D -   v  3Fo", Null, 1949.1428, 0., 9, 9, Null, 
    3.1421*^6, 0., 0.0017896}, {"Fe", "I", "37", "5D -   u  5Po", Null, 
    1934.5351, 0., 9, 7, Null, 2.5462*^7, 0., 0.011111}\
, {"Fe", "I", "", 
      "5D -  4p  3Do", Null, 1915.2235, 0., 9, 7, Null, 
    1.3696*^6, 0., 0.00058581}, 
   {"Fe", "I", "", "5D -  5p  5Fo", Null, 1888.4448, 0., 9, 9, Null, 
    5.9799*^6, 0., 0.0031971}, 
   {"Fe", "I", "", "5D -  5p  5Fo", Null, 1883.7787, 0., 9, 11, 
    Null, 8.1038*^6, 0., 0.0052694}, 
   {"Fe", "I", "", "5D -  5p  5Do", Null, 1875.1584, 0., 9, 9, Null, 
    1.6589*^7, 0., 0.008745}, 
   {"Fe", "I", "", "5D -  5p  5Do", Null, 1861.0335, 0., 9, 7, Null, 
    7.8079*^6, 0., 0.0031532}, 
   {"Fe", "I", "", "5D -      5Po", Null, 1851.688, 0., 9, 7, Null, 
    5.5579*^7, 0., 0.022221}, 
   {"Fe", "I", "", "5D -      5Fo", Null, 1851.3805, 0., 9, 11, 
    Null, 2.5105*^6, 0., 0.0015767}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2610.625, 2611.4061, 
    416.299, 30, 30, 1., 2.5146*^8, 0., 0.25709}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2631.3234, 2632.1081, 
    667.683, 6, 8, 0.0655, 6.0318*^7, 
    2.72*^8, 0.083531}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2631.048, 2631.8321, 
    862.613, 4, 6, 0.062549, 7.682*^7, 2.75*^8, 0.11966}\
, {"Fe", "II", "1", 
      "6D -   z  6Do", 2628.294, 2629.0777, 
    977.053, 2, 4, 0.046329, 8.5601*^7, 2.61*^8, 
    0.17741}, {"Fe", "II", "1", "6D -   z  6Do", 2625.668, 2626.4511, 
    384.79, 8, 10, 0.04522, 3.3528*^7, 2.7*^8, 0.043342}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2621.6694, 2622.4518, 
    977.053, 2, 2, 0.01302, 4.8495*^7, 2.66*^8, 0.05}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2620.4091, 2621.1912, 
    862.613, 4, 4, 0.00192, 3.5899*^6, 2.61*^8, 
    0.0036978}, {"Fe", "II", "1", "6D -   z  6Do", 
    2617.618, 2618.3991, 667.683, 6, 6, 
    0.034999, 4.3644*^7, 2.75*^8, 0.044859}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2613.825, 2614.6051, 
    862.613, 4, 2, 0.05288, 1.9875*^8, 2.66*^8, 0.10185}\
, {"Fe", "II", "1", 
      "6D -   z  6Do", 2611.8742, 2612.6542, 384.79, 8, 8, 
    0.11561, 1.0887*^8, 2.72*^8, 0.11141}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2607.088, 2607.8664, 
    667.683, 6, 4, 0.08754, 1.6577*^8, 2.61*^8, 
    0.11268}, {"Fe", "II", "1", "6D -   z  6Do", 2599.396, 2600.1729, 0., 
    10, 10, 0.28902, 2.2087*^8, 2.7*^8, 0.22387}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2598.37, 2599.1465, 384.79, 8, 
    6, 0.10251, 1.3072*^8, 2.75*^8, 0.099291}, 
   {"Fe", "II", "1", "6D -   z  6Do", 2585.8762, 2586.65, 0., 10, 8, 
    0.082929, 8.046*^7, 2.72*^8, 
    0.064565}, {"Fe", "II", "2", "6D -   z  6Fo", 
    2394.246, 2394.977, 416.299, 30, 42, 1., 
    2.8729*^8, 0., 0.34576}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2413.3112, 2414.045, 
    977.053, 2, 4, 0.0361, 1.0631*^8, 0., 
    0.18577}, {"Fe", "II", "2", "6D -   z  6Fo", 
    2411.069, 2411.802, 977.053, 2, 2, 
    0.040479, 2.3902*^8, 0., 0.20843}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2410.5202, 2411.2533, 
    862.613, 4, 6, 0.07534, 1.4842*^8, 0., 0.19406}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2406.6621, 2407.3942, 
    862.613, 4, 4, 0.054499, 1.6189*^8, 0., 
    0.14059}, {"Fe", "II", "2", "6D -   z  6Fo", 
    2404.887, 2405.6186, 667.683, 6, 8, 
    0.11377, 1.6928*^8, 0., 0.19582}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2404.4323, 2405.164, 
    862.613, 4, 2, 0.00942, 5.6082*^7, 0., 0.024319}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2399.2421, 2399.9728, 
    667.683, 6, 6, 0.06839, 1.3664*^8, 0., 0.11799}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2395.6262, 2396.3559, 384.79, 8, 10, 
    0.20623, 2.4834*^8, 2.9999*^8, 0.26725}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2395.4201, 2396.1497, 
    667.683, 6, 4, 0.0083, 2.5007*^7, 0., 
    0.014359}, {"Fe", "II", "2", "6D -   z  6Fo", 
    2388.6301, 2389.3582, 384.79, 8, 8, 0.068099, 
    1.0339*^8, 0., 0.088493}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2383.062, 2383.7884, 
    384.79, 8, 6, 0.00397, 8.0995*^6, 0., 0.0051759}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2382.039, 2382.765, 0., 10, 12, 
    0.28832, 2.9431*^8, 3.1*^8, 0.30061}, 
   {"Fe", "II", "2", "6D -   z  6Fo", 2373.737, 2374.4612, 0., 10, 
    10, 0.026949, 3.3344*^7, 2.9999*^8, 
    0.028184}, {"Fe", "II", "2", "6D -   z  6Fo", 2366.8674, 
    2367.5905, 0., 10, 8, 0.00015999, 237930., 0., 
    0.00015996}, {"Fe", "II", "3", "6D -   z  6Po", 
    2343.984, 2344.7035, 416.299, 30, 18, 1., 2.5358*^8, 
    3.6*^6, 0.1254}, {"Fe", "II", "3", "6D -   z  6Po", 2380.763, 
    2381.489, 667.683, 6, 8, 0.046279, 
    2.5197*^7, 2.68*^8, 0.028566}, 
   {"Fe", "II", "3", "6D -   z  6Po", 2364.8293, 2365.552, 384.79, 
    8, 8, 0.10925, 6.07*^7, 2.68*^8, 0.050923}, 
   {"Fe", "II", "3", "6D -   z  6Po", 2359.1064, 2359.828, 
    862.613, 4, 6, 0.061299, 4.5733*^7, 
    2.61*^8, 0.057272}, 
   {"Fe", "II", "3", "6D -   z  6Po", 2348.303, 2349.022, 667.683, 
    6, 6, 0.13164, 9.9591*^7, 2.61*^8, 0.082385}, 
   {"Fe", "II", "3", "6D -   z  6Po", 2344.283, 2345.001, 
    977.053, 2, 4, 0.07155, 8.162*^7, 0., 0.13458}, 
   {"Fe", "II", "3", "6D -   z  6Po", 2343.496, 2344.214, 0., 10, 8, 
    0.29149, 1.6636*^8, 2.68*^8, 0.10965}, 
   {"Fe", "II", "3", "6D -   z  6Po", 2338.0083, 2338.725, 
    862.613, 4, 4, 0.091939, 1.0571*^8, 0., 
    0.086684}, {"Fe", "II", "3", "6D -   z  6Po", 2332.801, 
    2333.516, 384.79, 8, 6, 0.14638, 1.1297*^8, 2.61*^8, 
    0.069169}, {"Fe", "II", "3", "6D -   z  6Po", 
    2327.397, 2328.111, 667.683, 6, 4, 
    0.050299, 5.8624*^7, 0., 0.031758}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2279.9162, 2280.6202, 384.79, 
    8, 10, Null, 3.8729*^6, 0., 0.0037749}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2267.5864, 2268.2878, 
    667.683, 6, 8, Null, 3.0879*^6, 0., 0.0031758}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2260.8601, 2261.56, 
    862.613, 4, 6, Null, 2.1241*^6, 0., 
    0.0024431}, {"Fe", "II", "4", "6D -   z  4Fo", 2260.081, 
    2260.7805, 0., 10, 10, Null, 4.8487*^6, 0., 0.0037154}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2255.9882, 2256.6869, 
    977.053, 2, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2253.1271, 2253.8254, 384.79, 
    8, 8, Null, 5.0724*^6, 0., 0.0038629}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2250.936, 2251.6338, 
    667.683, 6, 6, Null, 3.0974*^6, 0., 0.0023542}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2250.1762, 2250.8739, 
    862.613, 4, 4, Null, 1.5045*^6, 0., 0.0011427}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2240.346, 2241.0411, 
    667.683, 6, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "4", "6D -   z  4Fo", 2236.688, 2237.3823, 384.79, 8, 6, 
    Null, Null, Null, Null}, {"Fe", "II", "4", "6D -   z  4Fo", 
    2233.7531, 2234.4472, 0., 10, 8, Null, 403.38, 0., 
    2.4155*^-7}, {"Fe", "II", "5", "6D -   z  4Do", 2268.5644, 
    2269.2659, 977.053, 2, 4, Null, 549940., 0., 
    0.00084912}, {"Fe", "II", "5", "6D -   z  4Do", 
    2262.688, 2263.388, 862.613, 4, 4, Null, 
    1.8305*^6, 0., 0.0014059}, 
   {"Fe", "II", "5", "6D -   z  4Do", 2249.1794, 2249.8768, 0., 10, 8, Null, 
    4.1375*^6, 0., 0.0025119}, 
   {"Fe", "II", "", "6D -   z  8Po", Null, 1901.773, 0., 10, 8, Null, 
    232670., 0., 0.00010093}, {"Fe", "II", "", "6D -   z  8Po", Null, 
    1888.01, 0., 10, 10, Null, 21535., 0., 0.000011508}, 
   {"Fe", "II", "", "6D -   z  4Ho", Null, 1642.3706, 0., 10, 12, Null, 
    526.04, 0., 2.5527*^-7}, 
   {"Fe", "II", "", "6D -   z  4Ho", Null, 1639.628, 0., 10, 10, 
    Null, 160.57, 0., 6.4714*^-8}, 
   {"Fe", "II", "", "6D -   z  4Ho", Null, 1635.1402, 0., 10, 8, 
    Null, 142.87, 0., 4.5814*^-8}, 
   {"Fe", "II", "", "6D -   z  4Io", Null, 1625.6823, 0., 10, 10, Null, 
    76.572, 0., 3.0339*^-8}, {"Fe", "II", "", "6D -   z  4Io", Null, 
    1623.7136, 0., 10, 12, Null, 232.24, 0., 1.1015*^-7}, 
   {"Fe", "II", "", "6D -   y  4Do", Null, 1620.0608, 0., 10, 8, 
    Null, 76206., 0., 0.000023988}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1621.5144, 416.299, 
    30, 18, 1., 4.0443*^8, 0., 0.095652}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1639.4012, 977.053, 2, 
    4, 0.066679, 1.1741*^8, 0., 0.094617}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1636.3313, 
    862.613, 4, 4, 0.09271, 1.6419*^8, 0., 0.0659088}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1634.3498, 862.613, 4, 
    6, 0.0407, 4.8228*^7, 0., 0.028969}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1631.1285, 667.683, 6, 
    4, 0.061069, 1.0917*^8, 0., 0.02903}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1629.1596, 
    667.683, 6, 6, 0.12253, 1.4658*^8, 0., 
    0.058324}, {"Fe", "II", "8", "6D -   y  6Po", Null, 
    1625.9123, 667.683, 6, 8, 0.0165, 
    1.4889*^7, 0., 0.0078677}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1621.6856, 384.79, 8, 6, 
    0.16952, 2.0562*^8, 0., 0.060801}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1618.468, 384.79, 8, 8, 
    0.097139, 8.8888*^7, 0., 0.034907}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1608.4511, 0., 10, 8, 
    0.33318, 3.1063*^8, 0., 0.096383}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1621.6856, 384.79, 8, 6, 
    Null, 2.0562*^8, 0., 0.060801}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1618.468, 384.79, 8, 8, Null, 
    8.8888*^7, 0., 0.034907}, 
   {"Fe", "II", "8", "6D -   y  6Po", Null, 1608.4511, 0., 10, 8, 
    Null, 1.9964*^8, 0., 0.061944}, 
   {"Fe", "II", "", "6D -   z  2Go", Null, 1610.7441, 0., 10, 10, Null, 
    263.08, 0., 1.0233*^-7}, 
   {"Fe", "II", "", "6D -   z  2Go", Null, 1604.5587, 0., 10, 8, Null, 
    27186., 0., 8.3946*^-6}, 
   {"Fe", "II", "", "6D -   y  4Fo", Null, 1611.2005, 0., 10, 8, 
    Null, 714090., 0., 0.00022233}, 
   {"Fe", "II", "", "6D -   y  4Fo", Null, 1608.8005, 0., 10, 10, Null, 
    171.45, 0., 6.6527*^-8}, 
   {"Fe", "II", "", "6D -   z  2Io", Null, 1595.8573, 0., 10, 12, 
    Null, 328.08, 0., 1.5031*^-7}, 
   {"Fe", "II", "", "6D -   x  4Do", Null, 1588.6876, 0., 10, 8, Null, 
    409240., 0., 0.00012388}, 
   {"Fe", "II", "", "6D -   y  4Go", Null, 1565.5255, 0., 10, 12, Null, 
    39052., 0., 0.000017219}, 
   {"Fe", "II", "", "6D -   y  4Go", Null, 1563.7512, 0., 10, 10, 
    Null, 7530.28, 0., 2.7606*^-6}, 
   {"Fe", "II", "", "6D -   y  4Go", Null, 1561.5024, 0., 10, 8, Null, 
    4677.9, 0., 1.3677*^-6}, 
   {"Fe", "II", "", "6D -   z  2Fo", Null, 1555.5403, 0., 10, 8, 
    Null, 7401., 0., 2.1478*^-6}, 
   {"Fe", "II", "", "6D -   y  2Go", Null, 1542.4495, 0., 10, 10, 
    Null, 556.838, 0., 1.9861*^-7}, 
   {"Fe", "II", "", "6D -   z  2Ho", Null, 1529.9036, 0., 10, 12, Null, 
    262.81, 0., 1.1066*^-7}, 
   {"Fe", "II", "", "6D -   x  4Go", Null, 1524.8542, 0., 10, 12, Null, 
    384.15, 0., 1.6069*^-7}, 
   {"Fe", "II", "", "6D -   x  4Go", Null, 1522.1618, 0., 10, 10, Null, 
    16226., 0., 5.6364*^-6}, 
   {"Fe", "II", "", "6D -   x  4Go", Null, 1516.7295, 0., 10, 8, 
    Null, 345.33, 0., 9.528*^-8}, 
   {"Fe", "II", "", "6D -   x  4Fo", Null, 1514.8589, 0., 10, 10, 
    Null, 19925., 0., 6.85498*^-6}, 
   {"Fe", "II", "", "6D -   x  4Fo", Null, 1506.5395, 0., 10, 8, Null, 
    1167.1, 0., 3.1769*^-7}, 
   {"Fe", "II", "", "6D -   y  4Ho", Null, 1504.5846, 0., 10, 12, Null, 
    241.62, 0., 9.84017*^-8}, 
   {"Fe", "II", "", "6D -   y  2Ho", Null, 1481.1231, 0., 10, 12, Null, 
    205.01, 0., 8.091*^-8}, 
   {"Fe", "II", "", "6D -   w  4Do", Null, 1376.4269, 0., 10, 8, 
    Null, 541.448, 0., 1.2303*^-7}, 
   {"Fe", "II", "", "6D -   x  2Fo", Null, 1369.5601, 0., 10, 8, 
    Null, 471.94, 0., 1.0617*^-7}, 
   {"Fe", "II", "9", "6D -   x  6Po", Null, 1260.533, 0., 10, 8, 
    Null, 1.3129*^8, 0., 0.025003}, 
   {"Fe", "II", "", "6D -   v  4Do", Null, 1150.3555, 0., 10, 8, 
    Null, 7950.4, 0., 1.2618*^-6}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1148.3988, 416.299, 
    30, 42, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1154.3985, 977.053, 
    2, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1153.2719, 862.613, 
    4, 6, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1152.8755, 
    862.613, 4, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1151.1458, 667.683, 
    6, 8, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1150.6851, 667.683, 
    6, 6, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1150.4691, 977.053, 
    2, 2, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1150.2904, 667.683, 
    6, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1148.9564, 
    862.613, 4, 2, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1148.2773, 384.79, 8, 
    10, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1147.4092, 384.79, 8, 8, Null, 
    Null, Null, Null}, {"Fe", "II", "10", "6D -   y  6Fo", Null, 1146.9516, 
    384.79, 8, 6, Null, Null, Null, Null}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1144.9379, 0., 10, 12, 
    Null, 4.4504*^8, 0., 0.10495}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1143.226, 0., 10, 10, Null, 
    6.7901*^7, 0., 0.013305}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1142.3656, 0., 10, 8, Null, 
    3.1948*^7, 0., 0.0050003}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1148.2773, 384.79, 8, 
    10, Null, 3.1919*^8, 0., 0.078879}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1147.4092, 384.79, 8, 8, Null, 
    1.2783*^8, 0., 0.02523}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1146.9516, 384.79, 8, 6, Null, 
    4.6655*^7, 0., 0.006901}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1144.9379, 0., 10, 12, 
    Null, 4.4504*^8, 0., 0.10495}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1143.226, 0., 10, 10, Null, 
    6.7901*^7, 0., 0.013305}, 
   {"Fe", "II", "10", "6D -   y  6Fo", Null, 1142.3656, 0., 10, 8, Null, 
    3.1948*^7, 0., 0.0050003}, 
   {"Fe", "II", "11", "6D -      6Do", Null, 1133.665, 0., 10, 8, Null, 
    3.8912*^7, 0., 0.0059979}, 
   {"Fe", "II", "11", "6D -      6Do", Null, 1128.483, 0., 10, 10, 
    Null, 198.68, 0., 3.7932*^-8}, 
   {"Fe", "II", "", "6D -      6Do", Null, 1127.0984, 0., 10, 10, Null, 
    1.5748*^7, 0., 0.0029992}, 
   {"Fe", "II", "", "6D -      6Do", Null, 1125.4477, 0., 10, 8, Null, 
    7.2344*^7, 0., 0.010999}, 
   {"Fe", "II", "12-14", "6D -      6Po", Null, 1121.9748, 0., 10, 
    8, Null, 1.3246*^8, 0., 0.019999}, 
   {"Fe", "II", "", "6D -      4Go", Null, 1110.5832, 0., 10, 10, Null, 
    424.66, 0., 7.8524*^-8}, 
   {"Fe", "II", "", "6D -      4Go", Null, 1108.504, 0., 10, 12, 
    Null, 280860., 0., 0.000062087}, 
   {"Fe", "II", "", "6D -      6Fo", Null, 1112.048, 0., 10, 12, Null, 
    4.0994*^7, 0., 0.0091201}, {"Fe", "II", "", "6D -      6Fo", Null, 
    1110.2803, 0., 10, 10, Null, 6.0573*^6, 0., 
    0.0011194}, {"Fe", "II", "", "6D -      6Fo", Null, 1107.412, 
    0., 10, 8, Null, 12.777, 0., 1.8793*^-9}, 
   {"Fe", "II", "15,17", "6D -      4Fo", Null, 1106.3596, 0., 10, 
    10, Null, 8.1724*^6, 0., 0.0014997}, 
   {"Fe", "II", "15,17", "6D -      4Fo", Null, 1101.5567, 0., 10, 8, Null, 
    19277., 0., 2.8054*^-6}, {"Fe", "II", "", "6D -      4Do", Null, 
    1106.2208, 0., 10, 8, Null, 72006., 0., 
    0.000010568}, {"Fe", "II", "18", "6D -   w  6Po", Null, 
    1098.4486, 416.299, 30, 18, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1104.9693, 667.683, 
    6, 8, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1102.3836, 862.613, 
    4, 6, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1101.526, 384.79, 8, 8, Null, 
    Null, Null, Null}, {"Fe", "II", "18", "6D -   w  6Po", Null, 
    1100.5166, 977.053, 2, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1100.0198, 667.683, 
    6, 6, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1099.1323, 
    862.613, 4, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1096.8769, 0., 10, 8, Null, 
    2.2169*^8, 0., 0.031989}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1096.7824, 
    667.683, 6, 4, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1096.6073, 384.79, 8, 
    6, Null, Null, Null, Null}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1101.526, 384.79, 8, 8, Null, 
    3.2966*^7, 0., 0.0059967}, {"Fe", "II", "18", "6D -   w  6Po", Null, 
    1096.8769, 0., 10, 8, Null, 2.2169*^8, 0., 0.031989}, 
   {"Fe", "II", "18", "6D -   w  6Po", Null, 1096.6073, 384.79, 8, 
    6, Null, 1.5307*^8, 0., 0.020697}, 
   {"Fe", "II", "", "6D -      6Fo", Null, 1085.1381, 0., 10, 8, 
    Null, 1.0916*^6, 0., 0.00015417}, 
   {"Fe", "II", "", "6D -      6Fo", Null, 1083.4204, 0., 10, 10, Null, 
    2.3097*^7, 0., 0.0040644}, {"Fe", "II", "", "6D -      6Fo", Null, 
    1081.8748, 0., 10, 12, Null, 6.6467*^7, 0., 0.013996}, 
   {"Fe", "II", "", "6D -   x  4Ho", Null, 1085.903, 0., 10, 8, Null, 
    9109.9, 0., 1.2883*^-6}, 
   {"Fe", "II", "", "6D -   x  4Ho", Null, 1085.579, 0., 10, 10, 
    Null, 683630., 0., 0.00012078}, 
   {"Fe", "II", "", "6D -   x  4Ho", Null, 1084.992, 0., 10, 12, Null, 
    3.6992*^6, 0., 0.00078343}, 
   {"Fe", "II", "", "6D -   u  2Go", Null, 1084.9315, 0., 10, 10, 
    Null, 61708., 0., 0.000010889}, 
   {"Fe", "II", "", "6D -   u  2Go", Null, 1079.8821, 0., 10, 8, Null, 
    11023., 0., 1.5417*^-6}, {"Fe", "II", "", "6D -   v  4Fo", Null, 
    1083.63, 0., 10, 8, Null, 2906., 0., 4.0926*^-7}, 
   {"Fe", "II", "", "6D -   v  4Fo", Null, 1081.935, 0., 10, 10, Null, 
    178540., 0., 0.000031333}, {"Fe", "II", "", "6D -      4Do", Null, 
    1073.769, 0., 10, 8, Null, 8058.1, 0., 1.1143*^-6}, 
   {"Fe", "II", "", "6D -   u  4Fo", Null, 1069.695, 0., 10, 10, 
    Null, 18864., 0., 3.2359*^-6}, 
   {"Fe", "II", "", "6D -   u  4Fo", Null, 1069.66, 0., 10, 8, Null, 
    156880., 0., 0.000021528}, 
   {"Fe", "II", "19", "6D -      6Do", Null, 1063.9718, 0., 10, 8, Null, 
    3.3128*^7, 0., 0.0044978}, 
   {"Fe", "II", "19", "6D -      6Do", Null, 1063.1764, 0., 10, 10, Null, 
    3.5394*^8, 0., 0.059979}, 
   {"Fe", "II", "", "6D -   w  4Go", Null, 1063.002, 0., 10, 8, 
    Null, 23014., 0., 3.1189*^-6}, 
   {"Fe", "II", "", "6D -   w  4Go", Null, 1062.152, 0., 10, 10, 
    Null, 2.2479*^7, 0., 0.0038019}, 
   {"Fe", "II", "", "6D -   w  4Go", Null, 1061.685, 0., 10, 12, 
    Null, 134890., 0., 0.000027353}, 
   {"Fe", "II", "21", "6D -      6Po", Null, 1055.2617, 0., 10, 8, Null, 
    5.9887*^7, 0., 0.0079983}, 
   {"Fe", "II", "", "6D -      2Fo", Null, 1051.75, 0., 10, 8, 
    Null, 5107.7, 0., 6.7764*^-7}, 
   {"Fe", "II", "", "6D -      4Do", Null, 1039.313, 0., 10, 8, Null, 
    1.0903*^6, 0., 0.00014125}, 
   {"Fe", "II", "", "6D -   v  2Ho", Null, 1039.078, 0., 10, 10, 
    Null, 2212.3, 0., 3.581*^-7}, 
   {"Fe", "II", "", "6D -   t  2Fo", Null, 1037.808, 0., 10, 8, Null, 
    48286., 0., 6.2373*^-6}, 
   {"Fe", "II", "", "6D -      4F0", Null, 1014.859, 0., 10, 8, 
    Null, 302860., 0., 0.000037411}, 
   {"Fe", "II", "", "6D -      4F0", Null, 1014.233, 0., 10, 10, 
    Null, 1.7213*^6, 0., 0.00026546}, 
   {"Fe", "II", "", "6D -      4Do", Null, 1016.35, 0., 10, 8, Null, 
    137390., 0., 0.000017022}, 
   {"Fe", "II", "", "6D -      2Fo", Null, 1011.136, 0., 10, 8, Null, 
    1286.6, 0., 1.5776*^-7}, 
   {"Fe", "II", "", "6D -      2Go", Null, 1003.658, 0., 10, 8, Null, 
    1794.2, 0., 2.1677*^-7}, 
   {"Fe", "II", "", "6D -      2Go", Null, 1003.48, 0., 10, 10, Null, 
    30699., 0., 4.6345*^-6}, {"Fe", "II", "", "6D -      2Go", Null, 
    1002.435, 0., 10, 10, Null, 12079., 0., 1.8197*^-6}, 
   {"Fe", "II", "", "6D -      6Do", Null, 940.192, 0., 10, 10, Null, 
    9.0931*^7, 0., 0.01205}, 
   {"Fe", "II", "", "6D -      6Fo", Null, 939.165, 0., 10, 8, Null, 
    1.3822*^7, 0., 0.0014622}, 
   {"Fe", "II", "", "6D -      6Do", Null, 938.0118, 0., 10, 8, 
    Null, 1.7324*^7, 0., 0.0018281}, 
   {"Fe", "II", "", "6D -      6Fo", Null, 937.652, 0., 10, 12, 
    Null, 1.2356*^8, 0., 0.019543}, 
   {"Fe", "II", "22", "6D -      4Do", Null, 935.772, 0., 10, 8, Null, 
    3.0741*^6, 0., 0.00032285}, 
   {"Fe", "II", "", "6D -      6Fo", Null, 935.518, 0., 10, 12, Null, 
    2.3651*^7, 0., 0.0037239}, 
   {"Fe", "II", "23", "6D -        o", Null, 933.134, 0., 10, 8, Null, 
    1.0499*^6, 0., 0.00010965}, 
   {"Fe", "II", "", "6D -      4Go", Null, 928.325, 0., 10, 12, Null, 
    2.2938*^6, 0., 0.00035563}, 
   {"Fe", "II", "24", "6D -        o", Null, 926.898, 0., 10, 8, Null, 
    5.7278*^7, 0., 0.005902}, 
   {"Fe", "II", "25-27", "6D -      6Do", Null, 926.212, 0., 10, 
    10, Null, 3.4019*^8, 0., 0.043752}, 
   {"Fe", "II", "28", "6D -        o", Null, 923.879, 0., 10, 8, 
    Null, 1.8148*^7, 0., 0.0018578}, 
   {"Fe", "II", "", "6D -        o", Null, 921.501, 0., 10, 8, 
    Null, 5.5984*^6, 0., 0.00057016}, 
   {"Fe", "III", "", "5D -  4p  7Po", Null, 1233.512, 
    932.49, 3, 5, Null, 453820., 0., 
    0.00017254}, {"Fe", "III", "", "5D -  4p  7Po", Null, 
    1230.575, 738.9, 5, 5, Null, 548220., 0., 
    0.00012446}, {"Fe", "III", "", "5D -  4p  7Po", Null, 
    1226.008, 436.2, 7, 5, Null, 975410., 0., 
    0.00015799}, {"Fe", "III", "", "5D -  4p  7Po", Null, 
    1225.565, 738.9, 5, 7, Null, 60866., 0., 0.000019188}, 
   {"Fe", "III", "", "5D -  4p  7Po", Null, 1221.035, 436.2, 7, 7, Null, 
    512380., 0., 0.00011453}, {"Fe", "III", "", "5D -  4p  7Po", Null, 
    1214.566, 0., 9, 7, Null, 2.4843*^6, 0., 0.00042732}, 
   {"Fe", "III", "", "5D -  4p  7Po", Null, 1213.439, 436.2, 7, 9, 
    Null, 3889.2, 0., 1.1038*^-6}, 
   {"Fe", "III", "", "5D -  4p  7Po", Null, 1207.05, 0., 9, 9, 
    Null, 20251., 0., 4.4234*^-6}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1125.791, 
    422.896, 25, 15, 1., 6.8489*^8, 0., 0.078081}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1131.915, 738.9, 5, 7, 
    0.013339, 1.9249*^7, 0., 0.0051764}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1131.195, 
    932.49, 3, 5, 0.02983, 6.0409*^7, 0., 0.019314}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1130.402, 
    27.399, 1, 3, 0.039659, 1.3414*^8, 0., 
    0.077099}, {"Fe", "III", "1", "5D -  4p  5Po", Null, 
    1129.191, 932.49, 3, 3, 0.089298, 3.0303*^8, 0., 
    0.057927}, {"Fe", "III", "1", "5D -  4p  5Po", Null, 
    1128.724, 738.9, 5, 5, 0.11605, 2.3659*^8, 0., 0.045189}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1128.05, 436.2, 7, 7, 
    0.093628, 1.3658*^8, 0., 0.026056}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1126.729, 738.9, 5, 3, 
    0.069489, 2.3735*^8, 0., 0.027104}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1124.881, 436.2, 7, 5, 0.18628, 
    3.8367*^8, 0., 0.051988}, 
   {"Fe", "III", "1", "5D -  4p  5Po", Null, 1122.526, 0., 9, 7, 0.36246, 
    5.3668*^8, 0., 0.0788428}, 
   {"Co", "I", "4v", "4F -   z  4Fo", 3565.7612, 3566.7821, 
    793.084, 28, 28, 1., 1.3811*^7, 0., 0.026342}, 
   {"Co", "I", "4v", "4F -   z  4Fo", 3652.5373, 3653.578, 
    406.84, 6, 8, 0.019179, 862220., 0., 
    0.0023006}, {"Co", "I", "4v", "4F -   z  4Fo", 3647.659, 
    3648.698, 809.33, 4, 6, 0.017069, 1.0273*^6, 0., 
    0.0030757}, {"Co", "I", "4v", "4F -   z  4Fo", 3631.384, 
    3632.419, 816., 8, 10, 0.01413, 517310., 0., 
    0.0012791}, {"Co", "I", "4v", "4F -   z  4Fo", 
    3602.0773, 3603.105, 809.33, 4, 4, 
    0.10631, 9.9708*^6, 0., 0.019406}, 
   {"Co", "I", "4v", "4F -   z  4Fo", 3594.8642, 3595.89, 
    406.84, 6, 6, 0.14646, 9.2125*^6, 0., 0.017859}, 
   {"Co", "I", "4v", "4F -   z  4Fo", 3575.3542, 3576.375, 816., 8, 8, 
    0.20108, 9.6429*^6, 0., 0.018489}, 
   {"Co", "I", "4v", "4F -   z  4Fo", 3550.5844, 3551.599, 
    406.84, 6, 4, 0.040779, 3.9924*^6, 0., 
    0.00503338}, {"Co", "I", "4v", "4F -   z  4Fo", 
    3526.844, 3527.852, 0., 10, 10, 0.32168, 1.2856*^7, 0., 
    0.023988}, {"Co", "I", "4v", "4F -   z  4Fo", 
    3520.0744, 3521.081, 816., 8, 6, 0.06864, 4.5988*^6, 
    0., 0.0064108}, {"Co", "I", "4v", "4F -   z  4Fo", 3473.9702, 3474.965, 
    0., 10, 8, 0.06469, 3.3818*^6, 0., 0.0048978}, 
   {"Co", "I", "4v", "4F -   z  4Fo", 3526.844, 3527.852, 0., 10, 
    10, Null, 1.2856*^7, 0., 0.023988}, 
   {"Co", "I", "3", "4F -   x  4Do", 2534.3422, 2535.1048, 
    793.084, 28, 20, Null, Null, Null, Null}, 
   {"Co", "I", "3", "4F -   x  4Do", 2614.124, 2614.904, 
    406.84, 6, 8, Null, Null, Null, Null}, 
   {"Co", "I", "3", "4F -   x  4Do", 2594.1582, 2594.934, 
    809.33, 4, 6, Null, Null, Null, Null}, 
   {"Co", "I", "3", "4F -   x  4Do", 2574.348, 2575.119, 
    816., 8, 8, Null, 1.7356*^7, 0., 0.017255}, 
   {"Co", "I", "3", "4F -   x  4Do", 2567.343, 2568.112, 
    406.84, 6, 6, Null, 2.9975*^7, 0., 0.029638}, 
   {"Co", "I", "3", "4F -   x  4Do", 2562.123, 2562.891, 809.33, 
    4, 4, Null, 3.9321*^7, 0., 0.038729}, 
   {"Co", "I", "3", "4F -   x  4Do", 2544.253, 2545.017, 
    809.33, 4, 2, Null, 3.032*^8, 0., 0.14721}, 
   {"Co", "I", "3", "4F -   x  4Do", 2535.962, 2536.724, 406.84, 
    6, 4, Null, 1.921*^8, 0., 0.12355}, 
   {"Co", "I", "3", "4F -   x  4Do", 2528.967, 2529.727, 816., 8, 
    6, Null, 2.8174*^8, 0., 0.20273}, 
   {"Co", "I", "3", "4F -   x  4Do", 2521.3632, 2522.122, 
    0., 10, 8, Null, 3.0028*^8, 0., 0.22909}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2431.5234, 2432.2649, 
    793.084, 28, 28, Null, Null, Null, Null}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2473.9023, 2474.65, 816., 8, 10, Null, 
    Null, Null, Null}, {"Co", "I", "5", "4F -   x  4Fo", 2467.685, 
    2468.431, 406.84, 6, 8, Null, 7.018*^6, 0., 
    0.0085477}, {"Co", "I", "5", "4F -   x  4Fo", 2460.801, 2461.545, 
    809.33, 4, 6, Null, 1.1846*^7, 0., 0.016141}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2439.039, 2439.778, 
    809.33, 4, 4, Null, 2.6754*^8, 0., 0.23875}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2436.657, 2437.396, 406.84, 
    6, 6, Null, 2.6433*^8, 0., 0.23542}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2432.209, 2432.947, 
    816., 8, 8, Null, 2.5632*^8, 0., 0.22746}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2424.932, 2425.668, 
    0., 10, 10, Null, 3.1951*^8, 0., 0.28184}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2415.318, 2416.052, 406.84, 
    6, 4, Null, Null, Null, Null}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2402.062, 2402.793, 
    816., 8, 6, Null, 5.0648*^7, 0., 0.032878}, 
   {"Co", "I", "5", "4F -   x  4Fo", 2384.86, 2385.587, 0., 10, 8, Null, 
    2.4314*^7, 0., 0.016596}, {"Co", "I", "6", "4F -   x  4Go", 
    2408.62, 2409.3581, 793.084, 28, 36, Null, 
    Null, Null, Null}, {"Co", "I", "6", "4F -   x  4Go", 2415.29, 
    2416.024, 809.33, 4, 6, Null, 3.629*^8, 0., 0.47637}, 
   {"Co", "I", "6", "4F -   x  4Go", 2414.457, 2415.191, 
    406.84, 6, 8, Null, 3.3508*^8, 0., 0.3907}, 
   {"Co", "I", "6", "4F -   x  4Go", 2411.619, 2412.352, 
    816., 8, 10, Null, Null, Null, Null}, 
   {"Co", "I", "6", "4F -   x  4Go", 2407.251, 2407.983, 0., 10, 12, Null, 
    3.6446*^8, 0., 0.38019}, 
   {"Co", "I", "6", "4F -   x  4Go", 2392.027, 2392.756, 
    406.84, 6, 6, Null, 3.9646*^7, 0., 
    0.034029}, {"Co", "I", "6", "4F -   x  4Go", 2380.485, 
    2381.211, 816., 8, 8, Null, Null, Null, Null}, 
   {"Co", "I", "6", "4F -   x  4Go", 2365.0593, 2365.782, 0., 10, 
    10, Null, 1.2779*^7, 0., 0.010715}, 
   {"Co", "I", "6", "4F -   x  4Go", 2358.679, 2359.4, 
    816., 8, 6, Null, Null, Null, Null}, 
   {"Co", "I", "6", "4F -   x  4Go", 2335.108, 2335.824, 0., 10, 8, Null, 
    Null, Null, Null}, {"Co", "II", "1", "3F -   z  5Fo", 2202.9824, 
    2203.67, 0., 9, 9, Null, 733850., 0., 0.000534278}, 
   {"Co", "II", "", "3F -   z  5Do", 2158.172, 2158.85, 0., 9, 9, 
    Null, 221030., 0., 0.00015444}, 
   {"Co", "II", "", "3F -   z  5Do", 2125.2114, 2125.883, 
    0., 9, 7, Null, 164799.9999999999, 0., 0.000086848}, 
   {"Co", "II", "2", "3F -   z  5Go", 2111.445, 2112.114, 0., 9, 11, Null, 
    2.1843*^6, 0., 0.0017855}, 
   {"Co", "II", "2", "3F -   z  5Go", 2091.0541, 2091.719, 0., 9, 
    9, Null, 744550., 0., 0.00048838}, 
   {"Co", "II", "2", "3F -   z  5Go", 2076.135, 2076.797, 0., 9, 7, 
    Null, 64159.8, 0., 0.000032267}, 
   {"Co", "II", "3", "3F -   z  3Go", 2060.898, 2061.562, 697.15, 
    21, 27, 1., 7.2614*^7, 0., 0.0594868}, 
   {"Co", "II", "3", "3F -   z  3Go", 2065.544, 2066.204, 950.51, 
    7, 9, 0.11438, 2.4749*^7, 0., 0.020366}, 
   {"Co", "II", "3", "3F -   z  3Go", 2063.7822, 2064.442, 597.32, 
    5, 7, 0.10351, 2.8879*^7, 0., 0.025824}, 
   {"Co", "II", "3", "3F -   z  3Go", 2058.8122, 2059.471, 
    0., 9, 11, 0.09331, 1.6682*^7, 0., 
    0.012965}, {"Co", "II", "3", "3F -   z  3Go", 
    2036.5844, 2037.239, 950.51, 7, 7, 0.25776, 
    7.481*^7, 0., 0.046548}, 
   {"Co", "II", "3", "3F -   z  3Go", 2025.7543, 2026.407, 0., 9, 
    9, 0.42845, 9.8276*^7, 0., 0.060599}, 
   {"Co", "II", "3", "3F -   z  3Go", Null, 1998.539, 0., 9, 7, 0.0026, 
    799130., 0., 0.00037218}, 
   {"Co", "II", "4", "3F -   z  3Fo", 2018.799, 2019.4531, 697.15, 21, 21, 
    1., 1.0352*^8, 0., 0.063292}, 
   {"Co", "II", "4", "3F -   z  3Fo", 2050.739, 2051.396, 950.51, 7, 9, 
    0.00742, 1.7092*^6, 0., 0.0013864}, 
   {"Co", "II", "4", "3F -   z  3Fo", 2049.173, 2049.83, 597.32, 5, 
    7, 0.00126, 372910., 0., 0.00032887}, 
   {"Co", "II", "4", "3F -   z  3Fo", 2027.038, 2027.691, 597.32, 
    5, 5, 0.38654, 1.6602*^8, 0., 0.10234}, 
   {"Co", "II", "4", "3F -   z  3Fo", 2022.356, 2023.008, 
    950.51, 7, 7, 0.27428, 8.4733*^7, 0., 0.051988}, 
   {"Co", "II", "4", "3F -   z  3Fo", 2011.511, 2012.161, 
    0., 9, 9, 0.31467, 7.6838*^7, 0., 0.046649}, 
   {"Co", "II", "4", "3F -   z  3Fo", 2000.793, 2001.441, 950.51, 
    7, 5, 0.01544, 6.8943*^6, 0., 0.0029573}, 
   {"Co", "II", "4", "3F -   z  3Fo", Null, 1984.841, 0., 9, 7, 
    0.0004, 129299.9999999999, 0., 0.000059396}, 
   {"Co", "II", "5", "3F -   z  3Do", Null, 1949.7395, 697.15, 21, 15, 1., 
    1.6596*^8, 0., 0.06756}, 
   {"Co", "II", "5", "3F -   z  3Do", 2002.7534, 2003.402, 597.32, 5, 7, 
    0.00011, 37541., 0., 0.000031625}, 
   {"Co", "II", "5", "3F -   z  3Do", Null, 1977.774, 950.51, 7, 7, 0.00003, 
    10085., 0., 5.9143*^-6}, 
   {"Co", "II", "5", "3F -   z  3Do", Null, 1975.012, 597.32, 5, 5, 
    0.00592, 2.8381*^6, 0., 0.0016597}, 
   {"Co", "II", "5", "3F -   z  3Do", Null, 1957.428, 597.32, 5, 3, 0.19355, 
    1.5873*^8, 0., 0.054705}, 
   {"Co", "II", "5", "3F -   z  3Do", Null, 1950.1, 950.51, 7, 5, 
    0.32372, 1.6109*^8, 0., 0.065698}, 
   {"Co", "II", "5", "3F -   z  3Do", Null, 1941.28, 0., 9, 7, 
    0.47665, 1.7174*^8, 0., 0.075467}, 
   {"Co", "II", "", "3F -   y  5Do", Null, 1628.971, 0., 9, 9, 
    Null, 249500., 0., 0.000099256}, 
   {"Co", "II", "", "3F -   z  5Po", Null, 1578.669, 0., 9, 7, Null, 
    1.3197*^6, 0., 0.00038349}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1575.9133, 697.15, 21, 21, 1., 
    2.5737*^8, 0., 0.095824}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1598.468, 950.51, 7, 9, 
    0.03232, 1.8599*^7, 0., 0.00916018}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1593.297, 597.32, 5, 7, 0.02312, 
    1.7276*^7, 0., 0.0092051}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1577.045, 950.51, 7, 7, 0.28224, 
    2.1745*^8, 0., 0.081078}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1576.796, 597.32, 5, 5, 
    0.23526, 2.5388*^8, 0., 0.094639}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1574.545, 0., 9, 9, 0.35804, 
    2.1557*^8, 0., 0.080123}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1560.877, 950.51, 7, 5, 
    0.03028, 3.3685*^7, 0., 0.0087882}, 
   {"Co", "II", "", "3F -   y  3Fo", Null, 1553.754, 0., 9, 7, 
    0.038749, 3.1209*^7, 0., 0.00878537}, 
   {"Co", "II", "6", "3F -   y  3Do", Null, 1572.648, 0., 9, 7, 
    Null, 8.7053*^7, 0., 0.025105}, 
   {"Co", "II", "", "3F -   z  3Ho", Null, 1579.613, 0., 9, 11, 
    Null, 4.6844*^6, 0., 0.0021417}, 
   {"Co", "II", "", "3F -   z  3Ho", Null, 1567.573, 0., 9, 9, Null, 
    2.4013*^7, 0., 0.0088462}, 
   {"Co", "II", "", "3F -   z  1Go", Null, 1552.76, 0., 9, 9, Null, 
    4.1371*^7, 0., 0.014954}, 
   {"Co", "II", "", "3F -   y  3Go", Null, 1547.946, 0., 9, 11, Null, 
    5.5878*^7, 0., 0.024533}, 
   {"Co", "II", "", "3F -   y  3Go", Null, 1534.821, 0., 9, 9, 
    Null, 5.6857*^6, 0., 0.002008}, 
   {"Co", "II", "", "3F -   y  3Go", Null, 1534.333, 0., 9, 7, 
    Null, 1.6642*^6, 0., 0.00045683}, 
   {"Co", "II", "", "3F -   z  1Ho", Null, 1539.468, 0., 9, 11, 
    Null, 5.8886*^6, 0., 0.0025572}, 
   {"Co", "II", "", "3F -   z  1Fo", Null, 1514.743, 0., 9, 7, 
    Null, 5.6282*^6, 0., 0.0015058}, 
   {"Co", "II", "", "3F -   x  3Do", Null, 1480.951, 0., 9, 7, 
    Null, 1.8112*^8, 0., 0.046319}, 
   {"Co", "II", "", "3F -   x  3Go", Null, 1470.1849, 697.15, 21, 27, 1., 
    3.5574*^8, 0., 0.14821}, 
   {"Co", "II", "", "3F -   x  3Go", Null, 1475.812, 597.32, 5, 7, 
    0.25036, 3.3961*^8, 0., 0.15525}, 
   {"Co", "II", "", "3F -   x  3Go", Null, 1472.902, 950.51, 7, 9, 
    0.33167, 3.5201*^8, 0., 0.14729}, 
   {"Co", "II", "", "3F -   x  3Go", Null, 1466.203, 0., 9, 11, 
    0.40339, 3.5511*^8, 0., 0.13988}, 
   {"Co", "II", "", "3F -   x  3Go", Null, 1461.857, 950.51, 7, 7, 
    0.0060599, 8.4379*^6, 0., 0.0027033}, 
   {"Co", "II", "", "3F -   x  3Go", Null, 1452.566, 0., 9, 9, 
    0.00854, 9.4543*^6, 0., 0.0029906}, 
   {"Co", "II", "", "3F -   x  3Go", Null, 1441.823, 0., 9, 7, 0., 
    3525.5, 0., 8.5459*^-7}, 
   {"Co", "II", "", "3F -   w  3Do", Null, 1448.011, 0., 9, 7, 
    Null, 1.8471*^8, 0., 0.04516}, 
   {"Co", "II", "", "3F -   x  3Fo", Null, 1424.779, 0., 9, 9, Null, 
    5.7731*^7, 0., 0.017569}, 
   {"Co", "II", "", "3F -   x  3Fo", Null, 1419.287, 0., 9, 7, 
    Null, 1.3896*^6, 0., 0.00032641}, 
   {"Co", "II", "", "3F -   y  3Ho", Null, 1394.638, 0., 9, 11, Null, 
    313200., 0., 0.00011162}, {"Co", "II", "", "3F -   y  1Fo", Null, 
    1378.638, 0., 9, 7, Null, 1.5281*^6, 0., 0.00033865}, 
   {"Co", "II", "", "3F -   y  1Go", Null, 1367.106, 0., 9, 9, Null, 
    289930., 0., 0.000081238}, 
   {"Co", "II", "", "3F -   y  1Ho", Null, 1344.5, 0., 9, 11, 
    Null, 441190., 0., 0.00014614}, 
   {"Co", "II", "", "3F -   w  3Go", Null, 1203.217, 0., 9, 7, 
    Null, 190700., 0., 0.000032193}, 
   {"Co", "II", "", "3F -   w  3Go", Null, 1199.061, 0., 9, 9, 
    Null, 1.5425*^6, 0., 0.00033247}, 
   {"Co", "II", "", "3F -   w  3Go", Null, 1192.213, 0., 9, 11, 
    Null, 399980., 0., 0.00010417}, 
   {"Co", "II", "", "3F -   w  3Fo", Null, 1192.43, 0., 9, 7, Null, 
    342120., 0., 0.000056723}, 
   {"Co", "II", "", "3F -   w  3Fo", Null, 1188.49, 0., 9, 9, Null, 
    1.9227*^7, 0., 0.0040715}, {"Co", "II", "", "3F -   v  3Do", Null, 
    1187.417, 0., 9, 7, Null, 7.5655*^7, 0., 0.012438}, 
   {"Co", "II", "", "3F -   u  3Do", Null, 1149.672, 0., 9, 7, Null, 
    465480., 0., 0.000071739}, 
   {"Co", "II", "", "3F -   v  3Fo", Null, 1147.941, 0., 9, 9, 
    Null, 1.2649*^6, 0., 0.00024989}, 
   {"Co", "II", "", "3F -      5Fo", Null, 1041.92, 0., 9, 9, 
    Null, 2.0523*^6, 0., 0.00033401}, 
   {"Co", "II", "", "3F -      5Go", Null, 1039.105, 0., 9, 11, 
    Null, 1.6422*^6, 0., 0.00032491}, 
   {"Co", "II", "", "3F -      3Go", Null, 1031.542, 0., 9, 11, 
    Null, 1.4852*^6, 0., 0.00028957}, 
   {"Co", "II", "", "3F -      3Go", Null, 1023.487, 0., 9, 9, 
    Null, 2.8956*^6, 0., 0.00045473}, 
   {"Co", "II", "", "3F -      3Go", Null, 1017.515, 0., 9, 7, 
    Null, 2.8377*^6, 0., 0.00034258}, 
   {"Co", "II", "", "3F -      3Fo", Null, 1030.259, 0., 9, 9, Null, 
    1.9634*^7, 0., 0.0031243}, 
   {"Co", "II", "", "3F -      3Fo", Null, 1020.744, 0., 9, 7, Null, 
    9.4018*^6, 0., 0.0011422}, 
   {"Co", "II", "", "3F -      3Do", Null, 1025.678, 0., 9, 7, Null, 
    6.8869*^6, 0., 0.00084481}, 
   {"Co", "III", "", "4F -   z  6Do", Null, 1017.387, 0., 10, 10, 
    Null, 60558., 0., 9.3972*^-6}, 
   {"Co", "III", "", "4F -   z  6Do", Null, 1014.751, 0., 10, 8, 
    Null, 1.0798*^6, 0., 0.00013335}, 
   {"Co", "III", "", "4F -   z  6Fo", Null, 968.567, 0., 10, 12, Null, 
    18018., 0., 3.0409*^-6}, 
   {"Co", "III", "", "4F -   z  6Fo", Null, 967.24, 0., 10, 10, 
    Null, 8.7513*^6, 0., 0.0012274}, 
   {"Co", "III", "", "4F -   z  6Fo", Null, 966.165, 0., 10, 8, 
    Null, 1.0964*^7, 0., 0.0012274}, 
   {"Co", "III", "", "4F -   z  6Po", Null, 952.297, 0., 10, 8, Null, 
    7.3707*^7, 0., 0.0080168}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 942.7362, 818.228, 28, 20, 1., 
    1.0075*^9, 0., 0.095887}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 952.036, 451.4, 6, 8, 
    0.0040598, 9.9025*^6, 0., 0.0017941}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 951.59, 867.5, 4, 6, 0.00436, 
    1.4226*^7, 0., 0.0028969}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 948.499, 867.5, 4, 4, 
    0.047729, 2.3605*^8, 0., 0.031838}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 947.837, 451.4, 6, 6, 
    0.06508, 2.1504*^8, 0., 0.028963}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 946.608, 867.5, 4, 2, 
    0.10396, 1.0346*^9, 0., 0.069493}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 946.548, 841.5, 8, 8, 
    0.05913, 1.4716*^8, 0., 0.019766}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 944.7718, 451.4, 6, 4, 
    0.15632, 7.8249*^8, 0., 0.069799}, 
   {"Co", "III", "1", "4F -   z  4Do", Null, 942.389, 841.5, 8, 6, 0.22642, 
    7.6126*^8, 0., 0.076017}, {"Co", "III", "1", "4F -   z  4Do", Null, 
    939.061, 0., 10, 8, 0.33296, 8.48538*^8, 0., 
    0.089743}, {"Co", "III", "2", "4F -   z  4Fo", Null, 
    937.4222, 818.228, 28, 28, 1., 4.0075*^8, 0., 
    0.0527958}, {"Co", "III", "2", "4F -   z  4Fo", Null, 944.078, 
    841.5, 8, 10, 0.029949, 3.2895*^7, 0., 
    0.0054943}, {"Co", "III", "2", "4F -   z  4Fo", Null, 
    942.696, 451.4, 6, 8, 0.034339, 4.7349*^7, 0., 
    0.008411}, {"Co", "III", "2", "4F -   z  4Fo", Null, 
    941.748, 867.5, 4, 6, 0.025489, 4.7003*^7, 0., 
    0.0093743}, {"Co", "III", "2", "4F -   z  4Fo", Null, 
    938.645, 867.5, 4, 4, 0.098599, 
    2.7547*^8, 0., 0.036386}, 
   {"Co", "III", "2", "4F -   z  4Fo", Null, 938.072, 451.4, 6, 6, 0.11791, 
    2.2005*^8, 0., 0.02903}, {"Co", "III", "2", "4F -   z  4Fo", Null, 
    937.307, 841.5, 8, 8, 0.17668, 2.4791*^8, 0., 
    0.032652}, {"Co", "III", "2", "4F -   z  4Fo", Null, 936.637, 
    0., 10, 10, 0.32425, 3.6475*^8, 0., 
    0.047973}, {"Co", "III", "2", "4F -   z  4Fo", Null, 934.994, 
    451.4, 6, 4, 0.04208, 1.1898*^8, 0., 0.010396}, 
   {"Co", "III", "2", "4F -   z  4Fo", Null, 932.736, 841.5, 8, 6, 
    0.070979, 1.3473*^8, 0., 0.01318}, 
   {"Co", "III", "2", "4F -   z  4Fo", Null, 929.9728, 0., 10, 8, 
    0.079769, 1.1458*^8, 0., 0.011885}, 
   {"Ni", "I", "3v", "3F -   z  5Fo", 3437.275, 3438.26, 0., 9, 9, Null, 
    4.4384*^6, 0., 0.0078661}, 
   {"Ni", "I", "5v", "3F -   z  3Fo", 3480.0632, 3481.0622, 
    971.793, 21, 21, 1., 6.7019*^6, 0., 0.012175}, 
   {"Ni", "I", "5v", "3F -   z  3Fo", 3688.407, 3689.457, 
    216.519, 5, 7, 0.02676, 451980., 0., 0.0012913}, 
   {"Ni", "I", "5v", "3F -   z  3Fo", 3571.8601, 3572.88, 332.153, 
    7, 7, 0.27772, 5.1643*^6, 0., 0.00988338}, 
   {"Ni", "I", "5v", "3F -   z  3Fo", 3551.5263, 3552.541, 332.153, 
    7, 9, 0.010999, 161740., 0., 0.00039346}, 
   {"Ni", "I", "5v", "3F -   z  3Fo", 3519.759, 3520.765, 
    216.519, 5, 5, 0.15039, 4.0917*^6, 0., 
    0.0076038}, {"Ni", "I", "5v", "3F -   z  3Fo", 3413.472, 
    3414.451, 332.153, 7, 5, 0.12703, 3.7891*^6, 0., 
    0.0047304}, {"Ni", "I", "5v", "3F -   z  3Fo", 3409.572, 
    3410.55, 0., 9, 7, 0.017129, 365930., 0., 
    0.00049632}, {"Ni", "I", "5v", "3F -   z  3Fo", 
    3391.04, 3392.013, 0., 9, 9, 0.38998, 
    6.5915*^6, 0., 0.01137}, {"Ni", "I", "", "3F -        o", 3369.563, 
    3370.531, 0., 9, 7, Null, 1.8351*^7, 0., 0.024308}, 
   {"Ni", "I", "7v", "3F -   z  3Go", 3310.9753, 3311.9344, 
    971.793, 21, 27, 1., 4.958*^6, 0., 0.010483}, 
   {"Ni", "I", "7v", "3F -   z  3Go", 3380.8701, 3381.841, 
    216.519, 5, 7, 0.21202, 3.8084*^6, 0., 
    0.0091418}, {"Ni", "I", "7v", "3F -   z  3Go", 3371.9814, 
    3372.95, 332.153, 7, 9, 0.18418, 2.5935*^6, 0., 
    0.0056872}, {"Ni", "I", "7v", "3F -   z  3Go", 3282.6892, 
    3283.635, 332.153, 7, 7, 0.030459, 597499.9999999999, 0., 
    0.00096583}, {"Ni", "I", "7v", "3F -   z  3Go", 
    3232.931, 3233.864, 0., 9, 11, 0.55841, 
    7.2997*^6, 0., 0.013988}, 
   {"Ni", "I", "7v", "3F -   z  3Go", 3226.9792, 3227.911, 0., 9, 9, 0.014, 
    224940., 0., 0.00035136}, {"Ni", "I", "7v", "3F -   z  3Go", 
    3145.107, 3146.018, 0., 9, 7, 0.00094998, 21063., 
    0., 0.000024308}, {"Ni", "I", "11v", "3F -      3Fo", 3019.141, 3020.02, 
    0., 9, 7, Null, 6.4421*^6, 0., 0.0068511}, 
   {"Ni", "I", "12v", "3F -   y  3Do", 3035.729, 3036.6142, 
    971.793, 21, 15, 1., 5.9637*^6, 0., 0.0058888}, 
   {"Ni", "I", "12v", "3F -   y  3Do", 3195.5642, 3196.488, 
    216.519, 5, 7, 0.05249, 575040., 0., 0.0012332}, 
   {"Ni", "I", "12v", "3F -   y  3Do", 3129.3, 3130.207, 
    216.519, 5, 5, 0.03243, 529699.9999999999, 0., 
    0.00077809}, {"Ni", "I", "12v", "3F -   y  3Do", 
    3107.7104, 3108.612, 332.153, 7, 7, 
    0.011179, 133020., 0., 0.00019271}, 
   {"Ni", "I", "12v", "3F -   y  3Do", 3105.456, 3106.357, 
    216.519, 5, 3, 0.25563, 7.1206*^6, 0., 0.0061806}, 
   {"Ni", "I", "12v", "3F -   y  3Do", 3045.0031, 3045.889, 332.153, 7, 5, 
    0.15815, 2.8038*^6, 0., 0.0027855}, 
   {"Ni", "I", "12v", "3F -   y  3Do", 2984.1284, 2984.999, 0., 9, 
    7, 0.49013, 6.5941*^6, 0., 0.0068511}, 
   {"Ni", "I", "", "3F -      3Fo", 2347.5092, 2348.228, 0., 9, 9, 
    Null, 2.2306*^7, 0., 0.01844}, 
   {"Ni", "I", "", "3F -      3Fo", 2337.4853, 2338.202, 0., 9, 7, 
    Null, 3.4776*^7, 0., 0.02217}, 
   {"Ni", "I", "", "3F -      3Do", 2345.541, 2346.259, 
    0., 9, 7, Null, 2.2299*^8, 0., 0.14314}, 
   {"Ni", "I", "", "3F -      3Go", 2320.0312, 2320.744, 0., 9, 11, 
    Null, 6.9422*^8, 0., 0.68511}, 
   {"Ni", "I", "", "3F -      3Go", 2255.8742, 2256.573, 0., 9, 9, 
    Null, 1.2823*^7, 0., 0.0097894}, 
   {"Ni", "I", "", "3F -      3Fo", 2310.9582, 2311.669, 
    0., 9, 9, Null, 4.69958*^8, 0., 0.37649}, 
   {"Ni", "I", "", "3F -        o", 2289.984, 2290.69, 0., 9, 7, Null, 
    2.0859*^8, 0., 0.12757}, 
   {"Ni", "I", "", "3F -        o", 2254.8, 2255.498, 0., 9, 9, Null, 
    9.6253*^6, 0., 0.007341}, 
   {"Ni", "I", "", "3F -      3Do", 2026.6112, 2027.264, 0., 9, 7, 
    Null, 2.4279*^7, 0., 0.011635}, 
   {"Ni", "I", "18v", "3D -   z  3Po", 3528.993, 3530.0029, 
    731.453, 15, 9, 1., 1.0946*^8, 0., 0.12269}, 
   {"Ni", "I", "18v", "3D -   z  3Po", 3722.4863, 3723.545, 713.08, 
    3, 5, 0.0047799, 800310., 0., 0.0027725}, 
   {"Ni", "I", "18v", "3D -   z  3Po", 3610.4611, 3611.491, 
    879.813, 5, 5, 0.039369, 7.241*^6, 0., 
    0.014159}, {"Ni", "I", "18v", "3D -   z  3Po", 3597.6994, 
    3598.726, 713.08, 3, 3, 0.04503, 1.3955*^7, 0., 0.027094}, 
   {"Ni", "I", "18v", "3D -   z  3Po", 3524.5351, 3525.543, 204.786, 7, 5, 
    0.50646, 1.0017*^8, 0., 0.13332}, 
   {"Ni", "I", "18v", "3D -   z  3Po", 3510.332, 3511.336, 713.08, 
    3, 1, 0.11556, 1.1566*^8, 0., 0.071265}, 
   {"Ni", "I", "18v", "3D -   z  3Po", 3492.9533, 3493.953, 879.813, 5, 3, 
    0.28883, 9.7811*^7, 0., 0.10741}, 
   {"Ni", "I", "19v", "3D -   z  3Fo", 3451.186, 3452.1798, 
    731.453, 15, 21, 1., 5.9386*^7, 0., 0.14855}, 
   {"Ni", "I", "19v", "3D -   z  3Fo", 3515.05, 3516.055, 879.813, 5, 7, 
    0.25129, 4.2358*^7, 0., 0.10991}, 
   {"Ni", "I", "19v", "3D -   z  3Fo", 3458.4553, 3459.446, 
    713.08, 3, 5, 0.24715, 6.1258*^7, 0., 0.18318}, 
   {"Ni", "I", "19v", "3D -   z  3Fo", 3433.554, 3434.538, 
    204.786, 7, 7, 0.095469, 1.7271*^7, 0., 0.030542}, 
   {"Ni", "I", "19v", "3D -   z  3Fo", 3414.7604, 3415.74, 
    204.786, 7, 9, 0.38676, 5.5326*^7, 0., 0.12442}, 
   {"Ni", "I", "19v", "3D -   z  3Fo", 3361.551, 3362.517, 
    879.813, 5, 5, 0.01781, 4.8067*^6, 0., 0.0081476}, 
   {"Ni", "I", "19v", "3D -   z  3Fo", 3286.9421, 3287.889, 
    204.786, 7, 5, 0.0016399, 469180., 0., 
    0.00054313}, {"Ni", "II", "1", "2D -  4p  4Do", Null, 
    1939.569, 0., 6, 8, Null, 2562.4, 0., 1.9269*^-6}, 
   {"Ni", "II", "1", "2D -  4p  4Do", Null, 1896.15, 0., 6, 6, Null, 
    1.6798*^6, 0., 0.00090542}, 
   {"Ni", "II", "1", "2D -  4p  4Do", Null, 1864.467, 0., 6, 4, Null, 
    8830.3, 0., 3.068*^-6}, 
   {"Ni", "II", "", "2D -  4p  4Go", Null, 1842.889, 0., 6, 8, 
    Null, 1.3003*^6, 0., 0.00088277}, 
   {"Ni", "II", "", "2D -  4p  4Go", Null, 1817.564, 0., 6, 6, Null, 
    306210., 0., 0.00015165}, {"Ni", "II", "2", "2D -  4p  4Fo", Null, 
    1804.473, 0., 6, 8, Null, 1.0999*^7, 0., 0.0071589}, 
   {"Ni", "II", "2", "2D -  4p  4Fo", Null, 1783.318, 0., 6, 6, 
    Null, 2.4464*^6, 0., 0.0011664}, 
   {"Ni", "II", "2", "2D -  4p  4Fo", Null, 1772.28, 0., 6, 4, 
    Null, 289080., 0., 0.000090759}, 
   {"Ni", "II", "3", "2D -  4p  2Go", Null, 1773.949, 0., 6, 8, 1., 
    9.8894*^6, 0., 0.0062208}, 
   {"Ni", "II", "4", "2D -  4p  2Fo", Null, 1751.8289, 602.7768, 
    10, 14, 1., 1.4067*^8, 0., 0.090609}, 
   {"Ni", "II", "4", "2D -  4p  2Fo", Null, 1754.809, 
    506.94, 4, 6, 0.13261, 4.3304*^7, 0., 0.029987}, 
   {"Ni", "II", "4", "2D -  4p  2Fo", Null, 1751.91, 0., 6, 8, 0.42252, 
    1.04*^8, 0., 0.063804}, 
   {"Ni", "II", "4", "2D -  4p  2Fo", Null, 1709.6, 0., 6, 6, 
    0.44487, 1.5711*^8, 0., 0.068841}, 
   {"Ni", "II", "5", "2D -  4p  2Do", Null, 1744.2358, 
    602.7768, 10, 10, 1., 3.2299*^8, 0., 0.14732}, 
   {"Ni", "II", "5", "2D -  4p  2Do", Null, 1788.486, 
    506.94, 4, 6, 0.15053, 7.5166*^7, 0., 
    0.054068}, {"Ni", "II", "5", "2D -  4p  2Do", Null, 
    1748.282, 506.94, 4, 4, 0.37997, 
    3.0469*^8, 0., 0.13962}, 
   {"Ni", "II", "5", "2D -  4p  2Do", Null, 1741.549, 0., 6, 6, 
    0.4208, 2.2757*^8, 0., 0.10348}, 
   {"Ni", "II", "5", "2D -  4p  2Do", Null, 1703.405, 0., 6, 4, 
    0.048699, 4.2213*^7, 0., 0.012242}, 
   {"Ni", "II", "", "2D -  4p' 4Po", Null, 1536.939, 506.94, 4, 
    6, Null, 1.0732*^7, 0., 0.0057009}, 
   {"Ni", "II", "", "2D -  4p' 4Po", Null, 1536.741, 506.94, 4, 
    4, Null, 3.1614*^7, 0., 0.011193}, 
   {"Ni", "II", "", "2D -  4p' 4Po", Null, 1526.156, 
    506.94, 4, 2, Null, 4.7306, 0., 
    8.2592*^-10}, {"Ni", "II", "", "2D -  4p' 4Po", Null, 
    1502.148, 0., 6, 6, Null, 3.9316*^7, 0., 0.0133}, 
   {"Ni", "II", "", "2D -  4p' 4Po", Null, 1501.959, 0., 6, 4, 
    Null, 3.4101*^6, 0., 0.00076886}, 
   {"Ni", "II", "6", "2D -  4p''2Fo", Null, 1484.9721, 
    602.7768, 10, 14, 1., 5.2171*^7, 0., 0.024146}, 
   {"Ni", "II", "6", "2D -  4p''2Fo", Null, 1510.855, 
    506.94, 4, 6, 0.41653, 4.8144*^7, 0., 0.024714}, 
   {"Ni", "II", "6", "2D -  4p''2Fo", Null, 1477.222, 0., 6, 6, 
    0.026179, 3.2367*^6, 0., 0.0010589}, 
   {"Ni", "II", "6", "2D -  4p''2Fo", Null, 1467.756, 0., 6, 8, 
    0.55729, 5.2692*^7, 0., 0.022691}, 
   {"Ni", "II", "7", "2D -  4p''2Do", Null, 1472.742, 602.7768, 10, 
    10, 1., 1.9069*^8, 0., 0.062008}, 
   {"Ni", "II", "7", "2D -  4p''2Do", Null, 1500.434, 
    506.94, 4, 4, 0.31379, 1.4147*^8, 0., 0.047746}, 
   {"Ni", "II", "7", "2D -  4p''2Do", Null, 1487.452, 506.94, 4, 
    6, 0.00991, 3.0557*^6, 0., 0.0015203}, 
   {"Ni", "II", "7", "2D -  4p''2Do", Null, 1467.259, 0., 6, 4, 
    0.10714, 5.165*^7, 0., 0.011113}, 
   {"Ni", "II", "7", "2D -  4p''2Do", Null, 1454.842, 0., 6, 6, 
    0.56917, 1.8766*^8, 0., 0.059545}, 
   {"Ni", "II", "", "2D -  4p''2Po", Null, 1449.997, 0., 6, 4, Null, 
    1.6802*^7, 0., 0.0035306}, 
   {"Ni", "II", "", "2D -  4p' 4Do", Null, 1415.72, 0., 6, 6, 
    Null, 1.3742*^7, 0., 0.0041299}, 
   {"Ni", "II", "", "2D -  4p' 4Do", Null, 1414.292, 0., 6, 4, 
    Null, 1.1995*^7, 0., 0.002398}, 
   {"Ni", "II", "", "2D -  4p' 4Do", Null, 1412.866, 0., 6, 8, 
    Null, 1.6667*^7, 0., 0.0066504}, 
   {"Ni", "II", "", "2D -  4p' 2Do", Null, 1400.3663, 
    602.7768, 10, 10, 1., 1.5287*^8, 0., 0.044944}, 
   {"Ni", "II", "", "2D -  4p' 2Do", Null, 1423.206, 
    506.94, 4, 6, 0.07471, 1.8132*^7, 0., 
    0.0082592}, {"Ni", "II", "", "2D -  4p' 2Do", Null, 1411.065, 
    506.94, 4, 4, 0.56317, 2.1037*^8, 0., 
    0.062797}, {"Ni", "II", "", "2D -  4p' 2Do", Null, 1393.324, 0., 6, 6, 
    0.29522, 7.6364*^7, 0., 0.022225}, 
   {"Ni", "II", "", "2D -  4p' 2Do", Null, 1381.685, 0., 6, 4, 
    0.06691, 2.6624*^7, 0., 0.0050798}, 
   {"Ni", "II", "8", "2D -  4p' 2Po", Null, 1375.7289, 
    602.7768, 10, 6, 1., 7.6604*^8, 0., 0.13041}, 
   {"Ni", "II", "8", "2D -  4p' 2Po", Null, 1399.018, 506.94, 4, 
    4, 0.045189, 4.9367*^7, 0., 0.014486}, 
   {"Ni", "II", "8", "2D -  4p' 2Po", Null, 1381.286, 506.94, 4, 
    2, 0.35516, 8.0639*^8, 0., 0.11533}, 
   {"Ni", "II", "8", "2D -  4p' 2Po", Null, 1370.132, 0., 6, 4, 
    0.59966, 6.9752*^8, 0., 0.13087}, 
   {"Ni", "II", "", "2D -  4p' 4So", Null, 1345.878, 0., 6, 4, 1., 
    3.5487*^7, 0., 0.0064246}, 
   {"Ni", "II", "10", "2D - 4p'''2Fo", Null, 1324.1095, 602.7768, 
    10, 14, 1., 4.2422*^8, 0., 0.15611}, 
   {"Ni", "II", "10", "2D - 4p'''2Fo", Null, 1335.201, 506.94, 4, 
    6, 0.42189, 4.0729*^8, 0., 0.16328}, 
   {"Ni", "II", "10", "2D - 4p'''2Fo", Null, 1317.217, 0., 6, 8, 
    0.55758, 4.2047*^8, 0., 0.14583}, 
   {"Ni", "II", "10", "2D - 4p'''2Fo", Null, 1308.866, 0., 6, 6, 
    0.020549, 2.1048*^7, 0., 0.0054057}, 
   {"Ni", "II", "", "2D - 4p'''2Go", Null, 1252.771, 0., 6, 8, 1., 
    1.3283*^6, 0., 0.00041672}, 
   {"Ni", "II", "", "2D -(3Po) 4Fo", Null, 1055.9, 0., 6, 8, Null, 14415., 
    0., 3.2125*^-6}, 
   {"Ni", "II", "", "2D -(3Po) 4Do", Null, 1028.026, 0., 6, 6, Null, 10964., 
    0., 1.7372*^-6}, {"Ni", "II", "", "2D -(3Po) 2Go", Null, 1001.561, 0., 
    6, 8, 1., 33243., 0., 6.6657*^-6}, 
   {"Ni", "II", "", "2D -(3Po) 2Fo", Null, 1005.848, 0., 6, 8, 
    Null, 575430., 0., 0.00011637}, 
   {"Ni", "II", "", "2D -(3Po) 2Fo", Null, 993.947, 0., 6, 6, Null, 241690., 
    0., 0.000035797}, 
   {"Ni", "II", "", "2D -(3Po) 2Do", Null, 982.7558, 0., 6, 6, 
    Null, 1.3216*^6, 0., 0.00019136}, 
   {"Ni", "II", "", "2D -(3Po) 2Do", Null, 973.305, 0., 6, 4, Null, 
    116570., 0., 0.000011037}, 
   {"Ni", "II", "", "2D -  5p  4Do", Null, 956.908, 0., 6, 6, 
    Null, 2.2922*^6, 0., 0.00031467}, 
   {"Ni", "II", "", "2D -  5p  2Go", Null, 937.906, 0., 6, 8, 1., 
    569820., 0., 0.00010029}, 
   {"Ni", "II", "", "2D -  5p  4Go", Null, 947.876, 0., 6, 8, 
    Null, 806400., 0., 0.00014483}, 
   {"Ni", "II", "", "2D -  5p  4Go", Null, 940.883, 0., 6, 6, Null, 299860., 
    0., 0.000039797}, 
   {"Ni", "II", "", "2D -  5p  4Fo", Null, 946.353, 0., 6, 6, Null, 
    52952., 0., 7.1097*^-6}, 
   {"Ni", "II", "", "2D -  5p  2Fo", Null, 944.84, 0., 6, 8, 
    Null, 4.1052*^6, 0., 0.00073257}, 
   {"Ni", "II", "", "2D -  5p  2Fo", Null, 933.862, 0., 6, 6, Null, 
    4.4712*^6, 0., 0.00058459}, 
   {"Ni", "II", "", "2D -  5p    o", Null, 955.598, 0., 6, 8, 
    Null, 2.9957*^6, 0., 0.00054683}, 
   {"Ni", "II", "", "2D -  5p  2Do", Null, 944.633, 0., 6, 6, 
    Null, 1.3136*^7, 0., 0.0017573}, 
   {"Ni", "II", "", "2D -  5p  2Do", Null, 933.34, 0., 6, 4, 
    Null, 1.9815*^7, 0., 0.0017252}, 
   {"Ni", "II", "", "2D -(3Po) 4So", Null, 928.179, 0., 6, 4, 
    Null, 75652., 0., 6.514*^-6}, 
   {"Cu", "I", "1v", "2S -  4p  2Po", 3256.295, 3257.2345, 0., 2, 6, 1., 
    1.3512*^8, 0., 0.64476}, 
   {"Cu", "I", "1v", "2S -  4p  2Po", 3273.9543, 3274.898, 0., 2, 2, 
    0.33333, 1.3294*^8, 1.3469*^8, 0.21376}, 
   {"Cu", "I", "1v", "2S -  4p  2Po", 3247.5371, 3248.474, 0., 2, 4, 
    0.66667, 1.3622*^8, 1.3849*^8, 0.431}, 
   {"Cu", "I", "3", "2S -  4p' 2Po", 2179.873, 2180.5556, 0., 2, 6, 
    1., 9.1*^7, 0., 0.19469}, 
   {"Cu", "I", "3", "2S -  4p' 2Po", 2181.722, 2182.405, 0., 2, 2, 0.33333, 
    9.0768*^7, 0., 0.064813}, {"Cu", "I", "3", "2S -  4p' 2Po", 
    2178.9494, 2179.632, 0., 2, 4, 0.66667, 
    9.1116*^7, 0., 0.12979}, 
   {"Cu", "I", "4", "2S -  4p' 2Do", 2165.0952, 2165.775, 0., 2, 4, 1., 
    5.1*^7, 0., 0.071727}, 
   {"Cu", "I", "5", "2S -  5p  2Po", 2024.334, 2024.986, 0., 2, 6, 1., 
    9.8*^6, 0., 0.018074}, {"Cu", "I", "5", "2S -  5p  2Po", 
    2024.338, 2024.99, 0., 2, 4, 0.66667, 
    9.8*^6, 0., 0.012049}, {"Cu", "I", "5", "2S -  5p  2Po", 
    2024.326, 2024.978, 0., 2, 2, 0.33333, 9.8*^6, 0., 
    0.0060245}, {"Cu", "II", "1", "1S -  4p  3Po", Null, 
    1472.395, 0., 1, 3, 1., 8.5896*^6, 0., 0.0083753}, 
   {"Cu", "II", "2", "1S -  4p  3Do", Null, 1367.9509, 0., 1, 3, 1., 
    3.9435*^7, 0., 0.033189}, {"Cu", "II", "3", "1S -  4p  1Po", Null, 
    1358.773, 0., 1, 3, 1., 4.58*^8, 9.470*^8, 0.38031}, 
   {"Zn", "I", "1", "1S -  4p  1Po", 2138.573, 2139.2473, 0., 1, 3, 1., 
    7.09*^8, 7.09*^8, 1.4593}, 
   {"Zn", "I", "3", "1S -  5p  1Po", Null, 1589.561, 0., 1, 3, 1., 
    1.0727*^8, 0., 0.1219}, 
   {"Zn", "I", "4", "1S -  6p  1Po", Null, 1457.5717, 0., 1, 3, 1., 
    3.0322*^7, 0., 0.028973}, 
   {"Zn", "I", "", "1S -  7p  1Po", Null, 1404.1193, 0., 1, 3, 1., 
    1.2741*^7, 0., 0.011298}, 
   {"Zn", "II", "1", "2S -  4p  2Po", 2037.511, 2038.1674, 
    0., 2, 6, 1., 4.11*^8, 0., 0.76789}, 
   {"Zn", "II", "1", "2S -  4p  2Po", 2062.005, 2062.664, 0., 2, 2, 0.33333, 
    3.9653*^8, 3.9653*^8, 0.25292}, 
   {"Zn", "II", "1", "2S -  4p  2Po", 2025.4833, 2026.136, 0., 2, 4, 
    0.66667, 4.1837*^8, 4.1837*^8, 0.51497}, 
   {"Zn", "II", "2", "2S -  5p  2Po", Null, 984.9351, 0., 2, 6, 1., 
    6.7909*^6, 0., 0.0029629}, {"Zn", "II", "2", "2S -  5p  2Po", Null, 
    986.525, 0., 2, 2, 0.23985, 4.8629*^6, 0., 
    0.00070953}, {"Zn", "II", "2", "2S -  5p  2Po", Null, 
    984.142, 0., 2, 4, 0.76015, 7.7619*^6, 
    0., 0.0022541}, {"Zn", "II", "", "2S -(3Po) 4Po", Null, 
    949.463, 0., 2, 4, Null, Null, Null, Null}, 
   {"Zn", "II", "", "2S -(3Po) 4Po", Null, 938.713, 0., 2, 2, 
    Null, Null, Null, Null}, {"Zn", "II", "", "2S -(3Po) 4Fo", Null, 
    923.976, 0., 2, 4, Null, Null, Null, Null}, 
   {"Ga", "I", "1v", "2Po-  5s  2S ", 4124.636, 4125.8016, 
    550.793, 6, 2, 1., 1.41*^8, 0., 0.11994}, 
   {"Ga", "I", "1v", "2Po-  5s  2S ", 4172.043, 4173.219, 826.19, 4, 2, 
    0.66667, 9.0833*^7, 1.4111*^8, 
    0.11858}, {"Ga", "I", "1v", "2Po-  5s  2S ", 4032.9873, 
    4034.127, 0., 2, 2, 0.33333, 5.0277*^7, 1.4111*^8, 
    0.12267}, {"Ga", "I", "1", "2Po-  4d  2D ", 
    2920.1662, 2921.0233, 550.793, 6, 10, 1., 
    1.4999*^8, 0., 0.29847}, 
   {"Ga", "I", "1", "2Po-  4d  2D ", 2944.1733, 2945.034, 826.19, 4, 4, 
    0.066679, 2.2768*^7, 0., 0.029605}, 
   {"Ga", "I", "1", "2Po-  4d  2D ", 2943.6362, 2944.497, 826.19, 4, 
    6, 0.6, 1.3668*^8, 0., 0.26648}, 
   {"Ga", "I", "1", "2Po-  4d  2D ", 2874.2343, 2875.078, 
    0., 2, 4, 0.33333, 1.2235*^8, 0., 0.30324}, 
   {"Ga", "II", "2", "1S -  4p  1Po", Null, 1414.402, 0., 1, 3, 1., 2.*^9, 
    2.*^9, 1.7995}, {"Ga", "III", "", "2S -  4p  2Po", Null, 1507.957, 0., 
    2, 6, 1., 7.9*^8, 0., 0.80795}, 
   {"Ga", "III", "", "2S -  4p  2Po", Null, 1534.462, 0., 2, 2, 
    0.33333, 7.4976*^8, 7.4976*^8, 0.26466}, 
   {"Ga", "III", "", "2S -  4p  2Po", Null, 1495.045, 0., 2, 4, 
    0.66667, 8.1065*^8, 8.1065*^8, 0.54329}, 
   {"Ge", "I", "1", "3P -  5s  3Po", 2666.447, 2667.2443, 969.022, 
    9, 9, 1., 2.6731*^8, 0., 0.2851}, 
   {"Ge", "I", "1", "3P -  5s  3Po", 2754.588, 2755.402, 409.96, 
    5, 3, 0.15123, 1.1*^8, 0., 0.075123}, 
   {"Ge", "I", "1", "3P -  5s  3Po", 2709.624, 2710.427, 557.134, 3, 1, 
    0.12213, 2.8999*^8, 0., 0.10279}, 
   {"Ge", "I", "1", "3P -  5s  3Po", 2691.341, 2692.14, 
    557.134, 3, 3, 0.07822, 6.1*^7, 0., 0.06628}, 
   {"Ge", "I", "1", "3P -  5s  3Po", 2651.5683, 2652.3577, 0., 1, 3, 
    0.10423, 8.5*^7, 0., 0.26894}, 
   {"Ge", "I", "1", "3P -  5s  3Po", 2651.172, 2651.9613, 
    409.96, 5, 5, 0.40856, 2.*^8, 0., 
    0.21087}, {"Ge", "I", "1", "3P -  5s  3Po", 
    2592.534, 2593.3093, 557.134, 3, 5, 0.13563, 
    7.1*^7, 0., 0.11931}, 
   {"Ge", "I", "3", "3P -  4d  3Do", 2080.425, 2081.093, 969.022, 9, 15, 
    Null, Null, Null, Null}, {"Ge", "I", "3", "3P -  4d  3Do", 2105.824, 
    2106.4915, 409.96, 5, 5, Null, 1.7*^7, 0., 0.011309}, 
   {"Ge", "I", "3", "3P -  4d  3Do", 2102.258, 2102.9246, 
    409.96, 5, 3, Null, Null, Null, Null}, 
   {"Ge", "I", "3", "3P -  4d  3Do", 2094.2583, 2094.9237, 
    409.96, 5, 7, Null, 9.7999*^7, 0., 
    0.089358}, {"Ge", "I", "3", "3P -  4d  3Do", 
    2068.6562, 2069.3168, 557.134, 3, 5, Null, 1.2*^8, 
    0., 0.12839}, {"Ge", "I", "3", "3P -  4d  3Do", 2065.215, 
    2065.8746, 557.134, 3, 3, Null, 8.5*^7, 0., 0.054386}, 
   {"Ge", "I", "3", "3P -  4d  3Do", 2041.712, 2042.3676, 
    0., 1, 3, Null, 1.1*^8, 0., 0.20637}, 
   {"Ge", "II", "1", "2Po-  4p2 4P ", 2007.039, 2007.6883, 767.356, 
    4, 2, Null, Null, Null, Null}, 
   {"Ge", "II", "1", "2Po-  4p2 4P ", Null, 1979.2736, 767.356, 4, 
    4, Null, Null, Null, Null}, 
   {"Ge", "II", "1", "2Po-  4p2 4P ", Null, 1938.8906, 0., 2, 2, 
    Null, Null, Null, Null}, {"Ge", "II", "1", "2Po-  4p2 4P ", Null, 
    1938.0077, 767.356, 4, 6, Null, Null, Null, Null}, 
   {"Ge", "II", "1", "2Po-  4p2 4P ", Null, 1912.377, 0., 2, 4, Null, Null, 
    Null, Null}, {"Ge", "II", "2", "2Po-  5s  2S ", Null, 1633.3253, 
    178.237, 6, 2, 1., 9.9*^8, 0., 0.13198}, 
   {"Ge", "II", "2", "2Po-  5s  2S ", Null, 1649.1942, 767.356, 4, 2, 
    0.66667, 6.4113*^8, 9.9055*^8, 
    0.13071}, {"Ge", "II", "2", "2Po-  5s  2S ", Null, 
    1602.4863, 0., 2, 2, 0.33333, 3.4942*^8, 
    9.9055*^8, 0.13452}, 
   {"Ge", "II", "3", "2Po-  4p2 2D ", Null, 1563.9937, 
    178.237, 6, 10, Null, Null, Null, Null}, 
   {"Ge", "II", "3", "2Po-  4p2 2D ", Null, 1581.0698, 767.356, 4, 
    4, Null, Null, Null, Null}, 
   {"Ge", "II", "3", "2Po-  4p2 2D ", Null, 1576.8547, 767.356, 4, 6, Null, 
    Null, Null, Null}, {"Ge", "II", "3", "2Po-  4p2 2D ", Null, 
    1538.0907, 0., 2, 4, Null, Null, Null, Null}, 
   {"Ge", "II", "4", "2Po-  4d  2D ", Null, 1253.6971, 178.237, 6, 
    10, 1., 2.2*^9, 0., 0.864}, 
   {"Ge", "II", "4", "2Po-  4d  2D ", Null, 1264.7096, 767.356, 4, 
    4, 0.066679, 3.5719*^8, 0., 0.085652}, 
   {"Ge", "II", "4", "2Po-  4d  2D ", Null, 1261.9053, 767.356, 4, 6, 0.6, 
    2.1573*^9, 0., 0.77254}, {"Ge", "II", "4", "2Po-  4d  2D ", Null, 
    1237.0589, 0., 2, 4, 0.33333, 1.9083*^9, 0., 
    0.87561}, {"Ge", "II", "", "2Po-  4p2 2S ", Null, 1180.4661, 
    178.237, 6, 2, Null, Null, Null, Null}, 
   {"Ge", "II", "", "2Po-  4p2 2S ", Null, 1188.733, 767.356, 4, 2, Null, 
    Null, Null, Null}, {"Ge", "II", "", "2Po-  4p2 2S ", Null, 
    1164.272, 0., 2, 2, Null, Null, Null, Null}, 
   {"Ge", "II", "5", "2Po-  4p2 2P ", Null, 1104.0555, 
    178.237, 6, 6, Null, Null, Null, Null}, 
   {"Ge", "II", "5", "2Po-  4p2 2P ", Null, 1120.471, 767.356, 4, 
    2, Null, Null, Null, Null}, 
   {"Ge", "II", "5", "2Po-  4p2 2P ", Null, 1106.746, 767.356, 4, 4, Null, 
    Null, Null, Null}, {"Ge", "II", "5", "2Po-  4p2 2P ", Null, 
    1098.714, 0., 2, 2, Null, Null, Null, Null}, 
   {"Ge", "II", "5", "2Po-  4p2 2P ", Null, 1085.513, 0., 2, 4, 
    Null, Null, Null, Null}, {"Ge", "II", "6", "2Po-  6s  2S ", Null, 
    1068.3059, 178.237, 6, 2, 1., 2.*^8, 0., 0.011407}, 
   {"Ge", "II", "6", "2Po-  6s  2S ", Null, 1075.072, 767.356, 4, 
    2, 0.66667, 1.3083*^8, 0., 0.011335}, 
   {"Ge", "II", "6", "2Po-  6s  2S ", Null, 1055.0261, 0., 2, 2, 0.33333, 
    6.9215*^7, 0., 0.011559}, 
   {"Ge", "II", "7", "2Po-  5d  2D ", Null, 1010.752, 
    178.237, 6, 10, 1., 2.2*^8, 0., 0.056159}, 
   {"Ge", "II", "7", "2Po-  5d  2D ", Null, 1017.06, 767.356, 4, 
    4, 0.066679, 3.5999*^7, 0., 0.0055813}, 
   {"Ge", "II", "7", "2Po-  5d  2D ", Null, 1016.6377, 767.356, 4, 
    6, 0.6, 2.1629*^8, 0., 0.05025}, 
   {"Ge", "II", "7", "2Po-  5d  2D ", Null, 999.1011, 0., 2, 4, 0.33333, 
    1.8982*^8, 0., 0.056813}, 
   {"Ge", "III", "", "1S -  4p  3Po", Null, 1600.1, 0., 1, 3, 1., 
    Null, Null, Null}, {"Ge", "III", "", "1S -  4p  1Po", Null, 1088.5, 0., 
    1, 3, 1., 3.36*^9, 3.36*^9, 1.7905}, 
   {"Ge", "IV", "", "2S -  4p  2Po", Null, 1202.3277, 0., 2, 6, 1., 
    1.21*^9, 0., 0.78679}, 
   {"Ge", "IV", "", "2S -  4p  2Po", Null, 1229.84, 0., 2, 2, 
    0.33333, 1.1306*^9, 1.1306*^9, 0.25636}, 
   {"Ge", "IV", "", "2S -  4p  2Po", Null, 1189.028, 0., 2, 4, 
    0.66667, 1.2511*^9, 1.2511*^9, 0.53034}} 

Abbreviation[Deuterium] := "D";

FindIons[]:=
  (Message[FindIons::argrx, FindIons, 0, 2]; {})

FindIons[wavelength1_]:=
  (Message[FindIons::argr, FindIons, 2]; {})

FindIons[wavelength1_, wavelength2_, extra__]:=
  (Message[FindIons::argrx, FindIons, Length[{extra}]+2, 2]; {})

AtomicData[]:=
  (Message[AtomicData::argt, AtomicData, 0, 1, 2]; {})

AtomicData[ele_, ionlevel_, extra__]:=
  (Message[AtomicData::argt, AtomicData, Length[{extra}]+2, 1, 2]; {})

AtomicData[ele_] :=
  AtomicData[ele, _]
AtomicData[ele_, ionlevel_] :=
  Module[{airwav, vacwav, rel,
         Epl, GU, GL, A, gamma,
         f, a, b, elem, ionl},
  elem = If[(StringQ[ele]&&Length[Characters[ele]]<3), ele,
            ElementData[ToString[ele],"Abbreviation"]]; 
  Cases[prop, {elem,
             ionl:ionlevel,_,
             _, airwav_, vacwav_,
             Epl_, GL_, GU_, rel_,
             A_, gamma_, f_} :> DeleteCases[
         {
    Element -> elem,
    IonStage -> ionl, 
	AirWavelength -> airwav*Angstrom,
	VacuumWavelength -> vacwav*Angstrom,
	LowerTermFineStructureEnergy ->
	           Epl/Centimeter, 
	UpperStatisticalWeight -> GU, 
	LowerStatisticalWeight -> GL,
	RelativeStrength -> rel, 
	TransitionProbability -> A/Second, 
	DampingConstant -> gamma/Second, 
	OscillatorStrength -> f}, (_ -> Null) | ( _ -> Null*_)]]]

LowerTermFineStructureEnergy[stuff___]:= 
    Sort[Map[Reverse, 
             DeleteCases[{LowerTermFineStructureEnergy, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {LowerTermFineStructureEnergy, VacuumWavelength}->{},
                          {LowerTermFineStructureEnergy, _}], 
AtomicData::usage=
  "AtomicData[element, ionstage], gives the spectral data of the resonance \
absorption lines produced by a given element."; 

              {1}]]

VacuumWavelength[stuff___] := VacuumWavelength/. AtomicData[stuff] /.
		VacuumWavelength -> {}

AirWavelength[stuff___]:=
     Sort[Map[Reverse, 
             DeleteCases[{AirWavelength, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {AirWavelength, VacuumWavelength}->{},
                          {AirWavelength, _}], 
              {1}]]

UpperStatisticalWeight[stuff___]:=
    Sort[Map[Reverse, 
             DeleteCases[{UpperStatisticalWeight, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {UpperStatisticalWeight, VacuumWavelength}->{},
                          {UpperStatisticalWeight, _}], 
              {1}]]
		
LowerStatisticalWeight[stuff___]:=
    Sort[Map[Reverse, 
             DeleteCases[{LowerStatisticalWeight, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {LowerStatisticalWeight, VacuumWavelength}->{},
                          {LowerStatisticalWeight, _}], 
              {1}]]
              
RelativeStrength[stuff___]:=
    Sort[Map[Reverse, 
             DeleteCases[{RelativeStrength, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {RelativeStrength, VacuumWavelength}->{},
                          {RelativeStrength, _}], 
              {1}]]
              
TransitionProbability[stuff___]:=
     Sort[Map[Reverse, 
             DeleteCases[{TransitionProbability, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {TransitionProbability, VacuumWavelength}->{},
                          {TransitionProbability, _}], 
              {1}]]

DampingConstant[stuff___]:=
    Sort[Map[Reverse, 
             DeleteCases[{DampingConstant, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {DampingConstant, VacuumWavelength}->{},
                          {DampingConstant, _}], 
              {1}]]

OscillatorStrength[stuff___]:=
     Sort[Map[Reverse, 
             DeleteCases[{OscillatorStrength, 
                          VacuumWavelength} /. AtomicData[stuff] /. 
                          {OscillatorStrength, VacuumWavelength}->{},
                          {OscillatorStrength, _}], 
              {1}]]

findIons[wave1_, wave2_] :=
     Module[{ele, ionlevel, linenum, vacwav},
  Cases[prop, {ele_, ionlevel_, _, _,_,
               vacwav_, ___} /; wave1 <= vacwav < wave2
              -> {ele, ionlevel, vacwav*Angstrom}]]

FindIons[wavelength1_, wavelength2_] :=
  findIons[Convert[wavelength1, Angstrom]/Angstrom,
           Convert[wavelength2, Angstrom]/Angstrom]/.
           {"H"->"Hydrogen", "D"->"Deuterium", "Li"->"Lithium", 
            "Be"->"Beryllium", "B"->"Boron", "C"->"Carbon", 
            "N"->"Nitrogen", "O"->"Oxygen", "F"->"Fluorine", "Ne"->"Neon",
            "Na"->"Sodium", "Mg"->"Magnesium", "Al"->"Aluminium",
            "Si"->"Silicon", "P"->"Phosphorus", "S"->"Sulfur", 
            "Cl"->"Chlorine", "Ar"->"Argon", "K"->"Potasssium", 
            "Ca"->"Calcium", "Sc"->"Scandium", "Ti"->"Titanium",
            "V"->"Vanadium", "Cr"->"Chromium", "Mn"->"Manganese",
            "Fe"->"Iron", "Co"->"Cobalt", "Ni"->"Nickel", 
            "Cu"->"Copper", "Zn"->"Zinc", "Ga"->"Gallium", 
            "Ge"->"Germanium"};
          
       


WavelengthAbsorptionMap[wav1_, wav2_, opts___?OptionQ] := Module[{ions,wave1,wave2,k=4,nn=0,h=0.5},
  wave1 = Convert[wav1, Angstrom]/Angstrom;
  wave2 = Convert[wav2, Angstrom]/Angstrom;
  ions = Apply[{#3/Angstrom, #1<>#2}&, findIons[wave1, wave2], 1];
  ions = Sort[ions];
  If[NumberQ[wave1]==True && NumberQ[wave2]==True,
     Show[Graphics[{Line[{{wave1,1}, {wave2,1}}],
                    Apply[{Line[{{#1, 0}, {#1, 1}}],
                          Text[#2, {#1, 1 +
                                 Mod[nn++, k]/(k-1)*h}, {0,-1}]}&,
                        ions, 1]}],
       opts,
       PlotRange   -> {{wave1, wave2}, {0, 1+h*k/(k-1)}},
       AspectRatio -> 0.3,
       Ticks  -> {Automatic, None},
       Axes   -> {True, False}], {}]]

ElementAbsorptionMap[ele_, stuff___, opts___?OptionQ] := 
   Module[{ionlevel, lines}, 
    lines = VacuumWavelength[ele, stuff]/Angstrom; 
     Show[Graphics[{Thickness[0], (Line[{{#1, 0}, {#1, 1}}] & ) /@ lines}], 
      opts, 
      Frame -> True, 
      FrameTicks -> {Automatic, None}, 
      AspectRatio -> 0.3]]; 

End[]


EndPackage[]
