//////////////////////////////////////////////////////////////////////////////////////
//
//   J/Link source code (c) 1999-2000, Wolfram Research, Inc. All rights reserved.
//
//   Use is governed by the terms of the J/Link license agreement, which can be found at
//   www.wolfram.com/solutions/mathlink/jlink.
//
//   Author: Todd Gayley
//
//////////////////////////////////////////////////////////////////////////////////////

package com.wolfram.jlink;

import java.net.*;
import java.io.*;

/** 
 * This class loader does all the loading of classes requested via the Mathematica function LoadJavaClass[].
 * It also loads native libraries and resources used by classes that it loads.
 * The directories and files it searches for classes and other resources are provided
 * to it via the addLocations() method, which is called from Mathematica during every call to LoadJavaClass[].
 * The JLink Mathematica code manages the class path locations, including providing a default set
 * of extra dirs. Directories are automatically searched for jar or zip
 * files they contain, so you do not have to name jar files explicitly in addLocations(), although you
 * can if you want to limit the search to a specific jar file in a directory.
 * <p>
 * Most uses of this class are internal to J/Link, and most programmers will never deal with it directly.
 * One reason to use this class is if you want to load a class in Java code that is found in the special
 * set of extra locations that this class loader knows about (such as Java subdirectories of Mathematica 
 * application directories). Each KernelLink has a JLinkClassLoader instance associated with it, 
 * which you can retrieve using the link's {@link KernelLink#getClassLoader() getClassLoader()} method.
 * Here is an example of loading a class from a special J/Link-specific directory.
 * <pre>
 * Class cls = Class.forName("Some.class.that.only.JLink.can.find", ml.getClassLoader());</pre>
 * You can add directories and jar files to the search path using the addLocations() method:
 * <pre>
 * // This is equivalent to calling AddToClassPath["/some/dir", "/another/dir"] in Mathematica.
 * ml.getClassLoader().addLocations(new String[]{"/some/dir", "/another/dir"}, true);</pre>
 * If you using this class from a standalone Java program you should first call
 * the KernelLink method {@link KernelLink#enableObjectReferences() enableObjectReferences()} to
 * initialize the class loader with the special Mathematica-specific locations for Java classes.
 * <p>
 * Another advanced reason to use this class is if you need the class loader
 * to have a specific parent loader, because that parent loader has certain capabilities or
 * can load classes from certain locations that J/Link would otherwise not know about.
 * <pre>
 * KernelLink ml = MathLinkFactory.createKernelLink(...);
 * ml.discardAnswer();
 * ml.setClassLoader(new JLinkClassLoader(mySpecialParentClassLoader));
 * ml.enableObjectReferences();</pre>
 * The class has some static methods that are intended to be used from Mathematica code. Here is an
 * example that loads a resource from a Mathematica application's Java subdirectory:
 * <pre>
 * (* Mathematica code *)
 * LoadJavaClass["com.wolfram.jlink.JLinkClassLoader"];
 * JLinkClassLoader`getInstance[]@getResource["myImage.gif"];</pre>
 * When using the class from Java, you will always have a specific link in hand, so instead of calling
 * static methods you should obtain a JLinkClassLoader instance from the link using its getClassLoader()
 * method and then call instance methods.
 * 
 * @since 3.0
 * @see KernelLink#getClassLoader()
 * @see KernelLink#setClassLoader(JLinkClassLoader)
 */
 
public class JLinkClassLoader extends URLClassLoader {

	
	// URI class was introduced in Java 1.4. We want to use it if available.
	private static boolean jvmHasURIClass;
	static {
		try {
			Class.forName("java.net.URI");
			jvmHasURIClass = true;
		} catch (Throwable t) {
			jvmHasURIClass = false;
		}
	}

    ///////////////////////////////////  Static Interface  //////////////////////////////////
        
    //  Static interface intended to be used by Mathematica code.
     
	/** 
     * Retrieves the JLinkClassLoader instance being used by the currently-active link.
     * This method is intended to be called only from Mathematica, where the "active link"
	 * is well-defined (it is the link on which the call is coming in on).
     * 
     * @since 3.0
	 */
	public static JLinkClassLoader getInstance() {
		return StdLink.getLink().getClassLoader();
	}
		
    /** 
     * Loads the class with the given name using the JLinkClassLoader instance of the currently-active link.
     * This method is intended to be called only from Mathematica, where the "active link"
     * is well-defined (it is the link on which the call is coming in on).
     * 
     * @since 3.0
     */
    public static Class classFromName(String name) throws ClassNotFoundException {
        // This used to be getInstance().loadClass(name), but that broke for classes like "[I" in Java 6.
        // Sun says "that wasn't how loadClass() was supposed to work; use Class.forName() instead."
        // in http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6387908. Not really sure whether
        // 2nd arg should be true or false. False seems closer to the original implementation.
        return Class.forName(name, false, getInstance());
    }

		
	///////////////////////////////////  Instance Methods  //////////////////////////////////
        
	
    /** 
     * Constructs a new JLinkClassLoader instance. Only advanced programmers will need to use this.
     * 
     * @since 3.0
     */
    public JLinkClassLoader() {
        // In the absence of any specifically set parent to use, make this loader's parent be the loader that
        // loaded this class. Got it? This increases the likelihood that J/Link will be able to find
        // application-specific classes by ensuring that it can find any classes visible to the loader that
        // loaded J/Link itself. [Although this could backfire if the loader that loaded J/Link had less
        // capabilities than the "Application" class loader, which is the default parent for loaders created
        // without a specific parent.]
        // This feature was motivated by the needs of Tomcat and webM. Tomcat 4 uses a special
        // classloader to load webapp classes. If JLink.jar is in webM's web-inf/lib dir, it will be loaded
        // by the special classloader for web apps, and now J/Link will have access to all the classes that
        // servlets and other Java code in the web app have access to. If we didn't set the parent below,
        // then LoadClass calls from an MSP would fail if they asked for classes visible only to web apps. In
        // fact (if JLink.jar is in webM's web-inf/lib dir), LoadClass["com.wolfram.jlink.MathLink"] would fail.
        // For normal "insallable Java" uses, this parent-setting makes no difference, as the classloader
        // that loads J/Link itself is the "application" classloader, which would be the default parent anyway.
        this(JLinkClassLoader.class.getClassLoader());
    }
    
    /** 
     * Constructs a new JLinkClassLoader instance that has the given class loader as its parent loader.
     * Only advanced programmers will need to use this.
     * 
     * @since 3.0
     */
    public JLinkClassLoader(ClassLoader parent) {       
		super(new URL[0], parent);
	}
	

    /** 
     * Adds URLs, directories, and jar files in which this classloader will look for classes.
     * The elements of the locations array must be full paths to directories or jar or zip files,
     * or http URLs.
     * The searchForJars parameter determines whether directories will be automatically searched
     * for any jar files they contain. 
     * 
     * @since 3.0
     */
	public synchronized void addLocations(String[] locations, boolean searchForJars) {
		
		if (locations == null)
			return;
			
		URL[] existingLocs = getURLs();

		for (int i = 0; i < locations.length; i++) {
			String thisLocation = locations[i];
			if (thisLocation.toLowerCase().startsWith("http:")) {
				// Is an http URL.
				try {
					addIfNew(new URL(thisLocation), existingLocs);
				} catch (MalformedURLException e) {
					continue;
				}
			} else {
				// Is a file or dir spec.
				if (!thisLocation.toLowerCase().endsWith(".jar") && !thisLocation.toLowerCase().endsWith(".zip")) {
					// Is a directory. Make sure every directory name ends with a dir separator.
					String dirName = thisLocation.endsWith(File.separator) ? thisLocation : thisLocation + File.separator;
					try {
						addIfNew(fileToURL(new File(dirName)), existingLocs);
						if (searchForJars) {
							// Scan through dirs looking for jars and zips to add.
							String[] filesInDir = (new File(dirName)).list();
							if (filesInDir != null) {
								for (int j = 0; j < filesInDir.length; j++) {
									String lowerCaseFile = filesInDir[j].toLowerCase();
									if (lowerCaseFile.endsWith(".jar") || lowerCaseFile.endsWith(".zip")) {
										File f = new File(dirName, filesInDir[j]);
										addIfNew(fileToURL(f), existingLocs);
									}
								}
							}
						}
					} catch (Exception e) {
						continue;
					}
				} else {
					// Is a jar or zip file.
					try {
						addIfNew(fileToURL(new File(thisLocation)), existingLocs);
					} catch (Exception e) {
						continue;
					}
				}
			}
		}
	}


    /** 
     * Gives the set of locations in which this class loader will search for classes.
     * Only the locations known by this loader, not its parent loader, are returned. 
     * 
     * @since 3.0
     */
	public synchronized String[] getClassPath() {

		URL[] existingLocs = getURLs();
		String[] result = new String[existingLocs.length];
		for (int i = 0; i < existingLocs.length; i++) {
			if (existingLocs[i].getProtocol().equals("file")) {
				String fileString = existingLocs[i].getFile();
				if (jvmHasURIClass) {
					try {
						URI u = new URI(existingLocs[i].toString());
						fileString = u.getPath();
					} catch (Exception e) {}
				}
				result[i] = fileString;
			} else {
				result[i] = existingLocs[i].toString();
			}
		}
		return result;
	}
	
	
    /** 
     * Converts an array of bytes into an instance of class Class.
     * This method is effectively a public export of the protected ClassLoader method defineClass().
     * It exists mainly for future internal uses within J/Link applications.
     * 
     * @since 3.0
     */
    public Class classFromBytes(String className, byte[] bytes) {
        
        Class c = defineClass(className, bytes, 0, bytes.length);
        resolveClass(c);
        return c;
    }



	protected String findLibrary(String libName) {
		
		String platformSpecificName = System.mapLibraryName(libName);
		URL[] locs = getURLs();
		for (int i = 0; i < locs.length; i++) {
			if (locs[i].getProtocol().equals("file")) {
				String fileName = locs[i].getFile();
				boolean isJarOrZip = fileName.toLowerCase().endsWith(".zip") || fileName.toLowerCase().endsWith(".jar");
				if (!isJarOrZip) {
					// Treat fileName as a dir name. Look for the library in this dir and also dir/Libraries/$SystemID/.
					File f = new File(fileName, platformSpecificName);
					if (f.exists())
						return f.getAbsolutePath();
					f = new File(fileName, "Libraries" + java.io.File.separator + 
										Utils.getSystemID() + java.io.File.separator + platformSpecificName);
					if (f.exists())
						return f.getAbsolutePath();
				}
			}
		}
		return null;
	}


	private void addIfNew(URL u, URL[] existingLocs) {

		for (int j = 0; j < existingLocs.length; j++)
			if (existingLocs[j].equals(u))
				return;
		addURL(u);
	}


	private static URL fileToURL(File f) throws MalformedURLException {
	
		// Using toURI() ensures proper handling of spaces in path names. The toURL() method
		// is broken in that regard and will not be fixed by Sun for backwards compatibility reasons.
		if (jvmHasURIClass) {
			return f.toURI().toURL();
		} else {
			return f.toURL();
		}
	}

}

