(* :Title: Packer.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 6.0 *)
		     
(* :Copyright: Mathematica source code (c) 1999-2007, Wolfram Research, Inc. All rights reserved. *)

(* :Discussion: 
   Functionality for packing component files and dirs into a .paclet file, and unpacking.
	
   This file is a component of the PacletManager Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   PacletManager uses a special system wherein one package context (PacletManager`) has its
   implementation split among a number of .m files. Each component file has its own private
   context, and also potentially introduces public symbols (in the PacletManager` context) and
   so-called "package" symbols, where the term "package" comes from Java terminology,
   referring to symbols that are visible everywhere within the implementation of PacletManager,
   but not to clients.
*)


PackPaclet::usage = "PackPaclet[\"dir\"] creates a .paclet file from the contents of dir. The directory must contain a PacletInfo.m file."
UnpackPaclet::usage = "UnpackPaclet[\"file\"] unpacks the specified .paclet file into the same directory in which the .paclet file is located. UnpackPaclet[\"file\", \"dir\"] unpacks the file into the specified directory."


Begin["`Package`"]


End[]  (* `Package` *)



(* Current context will be PacletManager`. *)

Begin["`Packer`Private`"]


Options[PackPaclet] = {Verbose -> False}

Options[UnpackPaclet] = {Verbose -> False}


PackPaclet::notfound = "Could not find specified file or directory `1`."

(*
    PackPaclet creates a single .paclet file out of the component files
    and dirs that make up a paclet. The name of the .paclet file is
    determined automatically from info in the PacletInfo.m file.
    Returns the full path to the created .paclet file.
    
    PackPaclet["dir"]
		dir must contain a PacletInfo.m file. Entire contents of dir
		including subdirs (but not the dir itself) will be packed into
        the paclet file. Paclet file will be placed parallel to dir.
        
    PackPaclet["dir", "destDir"]
        Same as above, except .paclet file will be placed into destDir.
        
    PackPaclet[{"fileOrDir1", "fileOrDir2", ...}, "destDir"]
        Paclet will contain the given files and dirs. One of the supplied
        files must be a PacletInfo.m file, or a dir that contains a
        PacletInfo.m file at its top level.

        In the above form, each "fileOrDir1" can be a pair {"root", "child"}.
        See the PacletPacker JavaDocs for more info.
*)

PackPaclet[dir_String] :=
	Module[{fullPath = getFullPath[dir]},
		If[StringQ[fullPath],
			PackPaclet[{dir}, DirectoryName[fullPath]],
		(* else *)
			Message[PackPaclet::notfound, dir];
			$Failed
		]
	]

PackPaclet[dir_String, destDir_String] := PackPaclet[{dir}, destDir]

PackPaclet[components:{(_String | {_String, _String})..}, destDir:_String:Directory[]] :=
	JavaBlock[
		Module[{fullPairs, fullPaths, packer, pacletFile},
			InstallJava[];
			fullPairs = If[StringQ[#], {#, ""}, #]& /@ components;
			fullPaths = {getFullPath[#1], #2}& @@@ fullPairs;
			If[!MatchQ[fullPaths, {{_String, _String}..}],
				Message[PackPaclet::notfound, Last[#]]& /@
					Select[Thread[{First /@ fullPaths, First /@ components}], First[#] === Null &];
				Return[$Failed]
			];
			(* TODO: Verbose otpion. *)
			packer = JavaNew["com.wolfram.paclet.PacletPacker"];
			packer@addSourceLocation[##]& @@@ fullPaths;
			packer@setDestination[destDir];
			pacletFile = packer@pack[];
			If[JavaObjectQ[pacletFile],
				pacletFile@getAbsolutePath[],
			(* else *)
				(* Rely on error message from pack() method. *)
				$Failed
			]
		]
	]


UnpackPaclet[pacletFile_String] := UnpackPaclet[pacletFile, Automatic]

UnpackPaclet[pacletFile_String, destDir:(_String | Automatic)] :=
	JavaBlock[
		Module[{fullPath, fullDestPath, resultDir},
			InstallJava[];
			fullPath = getFullPath[pacletFile];
			If[!StringQ[fullPath],
				Message[UnpackPaclet::notfound, pacletFile];
				Return[$Failed]
			];
			If[destDir === Automatic,
				fullDestPath = DirectoryName[fullPath],
			(* else *)
				fullDestPath = getFullPath[destDir]
			];
			packer = JavaNew["com.wolfram.paclet.PacletPacker"];
			packer@addSourceLocation[fullPath];
			packer@setDestination[fullDestPath];
			(* TODO: Verbose option. *)
			resultDir = packer@unpack[];
			If[JavaObjectQ[resultDir],
				resultDir@getAbsolutePath[],
			(* else *)
				$Failed
			]
		]
	]
	
	
	
(* Returns the full path to the file or dir, or Null if it could not be found.
   fileOrDir can be a full or partial path.
*)
getFullPath[fileOrDir_String] :=
	Module[{f},
		Scan[
			Function[dir,
				f = ToFileName[{dir}, fileOrDir];
				(* Use FileInformation instead of just returning f to get the
                   full path even if the file is found in the current dir.
				*)
				If[FileType[f] =!= None, Return[File /. FileInformation[f]]]
			],
			(* "" is first entry, to handle case where fileOrDir is a full path.
			   Next check Directory[] before going to $Path. This allows people
			   to call SetDirectory[] and be sure that files will be found
			   in current dir first (current dir is NOT first on $Path).
			*)
			{"", Directory[]} ~Join~ $Path
		]
	]
	

End[]