(* :Title: Utils.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 6.0 *)
		     
(* :Copyright: Mathematica source code (c) 1999-2007, Wolfram Research, Inc. All rights reserved. *)

(* :Discussion: 
   Utility funcs.
	
   This file is a component of the PacletManager Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   PacletManager uses a special system wherein one package context (PacletManager`) has its
   implementation split among a number of .m files. Each component file has its own private
   context, and also potentially introduces public symbols (in the PacletManager` context) and
   so-called "package" symbols, where the term "package" comes from Java terminology,
   referring to symbols that are visible everywhere within the implementation of PacletManager,
   but not to clients.
*)

PacletNewerQ::usage = "PacletNewerQ is an internal symbol."


Begin["`Package`"]

DumpPaclets

preemptProtect
toCanonicalPath
log
$LogLevel
contextIndependentOptions

hasFrontEnd
hasLocalFrontEnd

userMessageFromException

(* Temporary only (?) *)
BuildPacletSiteFiles

(* Called outside of package (in paclet-int's build.xml file). *)
versionGreater

End[]  (* `Package` *)



(* Current context will be PacletManager`. *)

Begin["`Utils`Private`"]


PacletNewerQ[p1_Paclet, p2_Paclet] :=
	p1["Name"] == p2["Name"] && versionGreater[p1["Version"], p2["Version"]]
(* Also allow direct version number comparison:*)
PacletNewerQ[v1_String, v2_String] := versionGreater[v1, v2]


(* versionGreater compares two version specs and returns True if the first is
   greater (newer) than the second. False if equal or second is greater.
   Note that this provides comparisons for up to 5 digit blocks: 1.2.3.4.5.
*)
versionGreater[v1_String, v2_String] :=
	Not[
		OrderedQ[
			{PadRight[ToExpression /@ StringSplit[v1, "."], 5], PadRight[ToExpression /@ StringSplit[v2, "."], 5]}
		]
	]

(* splitVersion is a private utility for versionGreater. Takes "1.0.2" --> {1.0, 2}. *)
splitVersion[v_String] :=
	Block[{pos},
		pos = Last[Flatten[StringPosition[v, "."]]];
		ToExpression /@ {StringTake[v, pos - 1], StringDrop[v, pos]}
	]


(* Define a version of MathLink`PreemptProtect that is a no-op in 5.2 and earlier. *)

SetAttributes[preemptProtect, {HoldFirst}]

If[$VersionNumber >= 6,
	preemptProtect[e_] := MathLink`PreemptProtect[e],
(* else *)
	preemptProtect[e_] := e
]



(* Canonicalization handles expansion of . and .., conversions to correct case
   on Windows, etc. Canonicalizing paths allows us to use ==, MemberQ, etc. in
   M code without having to worry that two paths refer to the same place but are
   written slightly differently. It is also important when passing paths into Java,
   as we want to make sure that . and .. get resolved relative to Mathematica's
   current directory, not the JVM's.
*)
toCanonicalPath[path_String] :=
	(
		LoadJavaClass["com.wolfram.paclet.client.MUtils"];
		com`wolfram`paclet`client`MUtils`toCanonicalPath[path, Directory[]]
	)
	
	
(****************************  userMessageFromException  *******************************)

(**   Some Notes:

Unless otherwise stated, these strings are from getMessage[] (unwrapping HttpHandlerExceptions).

get this if network is down (e.g., cable modem off) and a proxy would be used:
   "java.net.SocketException: Can't connect to SOCKS proxy:connect timed out"
get this if network is down (e.g., cable modem off) and no proxy would be used:
   "org.apache.commons.httpclient.ConnectTimeoutException: The host did not accept the connection within timeout of 6000 ms"


*)

(* This signature is never deliberately called; it's a bug if it is. It exists just to prevent
   silly output to users in case of a bug elsewhere.
*)
userMessageFromException[Null, ___] = {"Unspecified error", ""}

userMessageFromException[exc_?JavaObjectQ, typeset:(True | False):False] :=
	Module[{innerException, statusCode, msgLines, firstLine},
		msgLines = {};
		firstLine = exc@getMessage[];
		innerException = exc@getCause[];
		Which[
			InstanceOf[exc, "com.wolfram.paclet.client.net.HttpHandlerException"],
				statusCode = exc@httpStatusCode;
				(* HttpHandlerExceptions with status code != SC_OK (200) are server errors,
				   including, of course 404 (file not found). SC_OK codes are other
				   types of errors, like connection timeout. They always wrap another
				   exception object.
				*)
				(* TODO: should I report full paclet URL? (Site name and paclet name are already listed via PSE ctor) *)
				Switch[statusCode,
					404,
					    (* TODO: Do I want to suggest that users update paclet site info here, and give the ma button? *)
						AppendTo[msgLines, "File not found on server."],
					200,
						AppendTo[msgLines, "Could not access server."];
						(*msgLines = msgLines ~Join~ userMessageFromException[innerException, typeset];*)
						If[typeset,
							msgLines = msgLines ~Join~ 
								{"\n",
								 Row[{"Use the ", Button[Style["Help \[FilledRightTriangle] Internet Connectivity", "MenuName"],
									FrontEnd`SetOptions[FrontEnd`$FrontEnd, FrontEnd`PreferencesSettings->{"Page"->"InternetConnectivity"}]; 
									FrontEnd`FrontEndToken["PreferencesDialog"], Evaluator->None, Method->"Preemptive",
									BaseStyle->"Link", Appearance->Frameless], " menu item to"}
								],  (* Line wraps right after button if we don't break manually. *)
								"test or configure Internet connectivity."
								}
						],
					_,
						(* TODO: Expand this handling to give sensible messages for other HTTP codes. *)
						AppendTo[msgLines, "Server returned HTTP status code " <> ToString[statusCode] <> "."]
				],
			InstanceOf[exc, "com.wolfram.paclet.client.PacletSiteException"],
				If[firstLine === Null, firstLine = "Error in paclet site"];
				AppendTo[msgLines, firstLine];
				If[innerException =!= exc && innerException =!= Null,
					msgLines = msgLines ~Join~ userMessageFromException[innerException, typeset]					
				],
			InstanceOf[exc, "com.wolfram.paclet.PacletInfoException"],
				If[firstLine === Null, firstLine = "Error in PacletInfo.m information"];
				AppendTo[msgLines, firstLine];
				If[innerException =!= exc && innerException =!= Null,
					msgLines = msgLines ~Join~ userMessageFromException[innerException, typeset]					
				],
			InstanceOf[exc, "com.wolfram.paclet.PacletException"],
				If[firstLine === Null, firstLine = "Paclet error"];
				AppendTo[msgLines, firstLine];
				If[innerException =!= exc && innerException =!= Null,
					msgLines = msgLines ~Join~ userMessageFromException[innerException, typeset]					
				],
			True,
				AppendTo[msgLines, exc@toString[]]
		];
		msgLines
	]
	
	
(******************************  debugging utils  *********************************)

DumpPaclets[] := getPacletManager[]@getPacletExpressions[]

DumpPacletObjects[] :=
	Module[{pm, iter, coll, paclets},
		pm = getPacletManager[];
		coll = pm@getPaclets[];
		iter = coll@iterator[];
		paclets = {};
		While[iter@hasNext[],
			AppendTo[paclets, iter@next[]]
		];
		paclets
	]


DumpPacletSites[] :=
	JavaBlock[
		Module[{siteObjects},
			siteObjects = JavaObjectToExpression[getPacletManager[]@getPacletSiteManager[]@getPacletSiteExpressions[]];
			{#@getBaseURL[], #@getPacletExpressions[]}& /@ siteObjects
		]
	]
	

(********************************  log  ***********************************)

(* Simple logging scheme intended mainly for debugging. Set PacletManager`Package`$LogLevel
   to an integer value to turn on logging. The first arg in the call to log is the
   "log level". If it is <= $LogLevel the message gets logged. Use the following as
   a general guide for log levels in calls to log:
   
   1   Information that might conceivably be useful to users. That is, assume that
       it isn't out of the question that a user might want to turn on logging at this level.
   
   2   This information is only for debugging. This would be a typical choice for $LogLevel
       if a developer was trying to follow what was happening internally.
   
   3   Verbose tracing-type information. A developer wants to know as much as possible
       about what was going on.
*)

(* So args are not evaled if they are not going to be logged. *)
SetAttributes[log, HoldRest]

log[level_Integer, args___] :=
	If[level <= $LogLevel, Print[args]]
	
	
(***********************  contextIndependentOptions  *************************)

(* This processes the names of options in a context-independent way. Also allows strings instead
   of symbols for LHS.
*)
contextIndependentOptions[optName:(_String | _Symbol), opts_List, defaults_List] :=
    First[ contextIndependentOptions[{optName}, opts, defaults] ]

contextIndependentOptions[optNames_List, opts_List, defaults_List] :=
    Module[{optNameStrings, stringifiedOptionSettings, stringifiedOptionDefaults},
        optNameStrings = (# /. x_Symbol :> SymbolName[x])& /@ optNames;
        stringifiedOptionSettings = MapAt[(# /. x_Symbol :> SymbolName[x])&, #, {1}]& /@ Flatten[{opts}];
        stringifiedOptionDefaults = MapAt[(# /. x_Symbol :> SymbolName[x])&, #, {1}]& /@ Flatten[{defaults}];
        optNameStrings /. stringifiedOptionSettings /. stringifiedOptionDefaults
    ]


(****************************************************************************)


(** Utility funcs for creation of a doc paclet site. These are not public, and temporary. *)

Options[BuildPacletSiteFiles] = {Verbose->False, "WRI"->False}

(* Call this on he top-level paclet site dir (the dir in which the PacletSite.m file will
   appear; this dir will have a Paclets dubdir holding a flat or subdir structure of .paclet files.
   
	For example:
	
	BuildPacletSiteFile["d:\\webserver\\tomcat5\\webapps\\pacletserver\\testsite"]
	
*)
BuildPacletSiteFiles[dir_String, OptionsPattern[]] :=
	JavaBlock[
		Module[{files, vbose, wri, englishSet, japaneseSet,
					englishDocNames, japaneseDocNames, docsWithNoJapaneseVersion},
			files = FileNames["*.paclet", ToFileName[dir, "Paclets"], Infinity];
			vbose = OptionValue[Verbose];
			wri = OptionValue["WRI"];
			InstallJava[];
			LoadJavaClass["com.wolfram.paclet.PacletFactory"];
			If[wri,
				(* On WRI server we separate English and Japanese sets into separate PacletSite files.
				   Special server logic decides which to serve to clients. Third-party paclet servers
				   will not have such logic, so for those the code just lumps all paclets into a single
				   PacletSite.m/mz/bin set.
				*)
				englishSet = Select[files, !StringMatchQ[#, "*SystemDocs_Japanese_*"]&];
				Print["length of englishSet: ", Length[englishSet]];
				buildSiteFiles[englishSet, dir, "", vbose];		
				(* Rather than including all English and Japanese docs in Japanese set, only
				   include the English versions of docs that are not present in Japanese form.
				*)
				englishDocNames = Flatten @ StringCases[files, ___ ~~ "SystemDocs_English_" ~~ name__ ~~ "-" ~~ __ :> name];
				Print["length of englishDocNames: ", Length[englishDocNames]];
				japaneseDocNames = Flatten @ StringCases[files, ___ ~~ "SystemDocs_Japanese_" ~~ name__ ~~ "-" ~~ __ :> name];
				Print["length of japaneseDocNames: ", Length[japaneseDocNames]];
				docsWithNoJapaneseVersion = englishDocNames ~Complement~ japaneseDocNames;
				docsWithNoJapaneseVersion = Flatten[
						Function[{shortName}, Select[englishSet, StringMatchQ[#, ___ ~~ "SystemDocs_English_" ~~ shortName ~~ __]&]] /@ docsWithNoJapaneseVersion
					];
				Print["length of docsWithNoJapaneseVersion: ", Length[docsWithNoJapaneseVersion]];
				japaneseSet = Join[
								Select[files, !StringMatchQ[#, "*SystemDocs_*"]&],
								Select[files, StringMatchQ[#, "*SystemDocs_Japanese_*"]&],
								docsWithNoJapaneseVersion
							  ];
				Print["length of japaneseSet: ", Length[japaneseSet]];
				buildSiteFiles[japaneseSet, dir, "_Japanese", vbose],	
			(* else *)
				buildSiteFiles[files, dir, "", vbose]			
			]
		]
	]

(* Worker function. *)
buildSiteFiles[files:{___String}, outDir_String, languageSuffix_String, verbose:(True | False)] :=
	Module[{siteFile, siteBinFile, paclets, pis, strm},
		paclets =
			Function[{filename},
				If[verbose, Print[filename]];
				First[com`wolfram`paclet`PacletFactory`createPaclets[filename]]
			] /@ files;
		pis = #@getPacletInfo[]@toString[]& /@ paclets;
		siteFile = ToFileName[outDir, "PacletSite" <> languageSuffix <> ".m"];
		siteBinFile = ToFileName[outDir, "PacletSite" <> languageSuffix <> ".bin"];
		strm = OpenWrite[siteFile];
		WriteString[strm, "PacletSite["];
		WriteString[strm, First[pis]];
		(WriteString[strm, ","]; WriteString[strm, #])& /@ Rest[pis];
		WriteString[strm, "]"];
		Close[strm];
		LoadJavaClass["com.wolfram.paclet.Utils"];
		com`wolfram`paclet`Utils`writePacletSiteBinFile[paclets, siteBinFile];
		compressPacletSiteFile[siteFile, languageSuffix];
	]

(* Compresses the given PacletSite.m file into PacletSite.mz in the same directory.
   PacletSite.m must be specified as a full path.
   Called automsticslly by BuildPacletSiteFile, so probsbly rarely used directly.
*)
compressPacletSiteFile[file_String, languageSuffix_String] :=
	JavaBlock[
		Module[{dir, fis, numRead, buf, zipStrm},
			dir = DirectoryName[file];
			fis = JavaNew["java.io.FileInputStream", file];
			zipStrm = JavaNew["java.util.zip.ZipOutputStream",
						JavaNew["java.io.FileOutputStream",
							ToFileName[dir, "PacletSite" <> languageSuffix <> ".mz"]]];
			(* In a compressed .mz file, the entry is always called PacletSite.m, even though the
			   actual filename might have a language suffix: PacletSite_Japanese.mz.
			*)
			zipStrm@putNextEntry[JavaNew["java.util.zip.ZipEntry", "PacletSite.m"]];
			buf = JavaNew["[B", 100000];
			While[(numRead = fis@read[buf]) >= 0,
				zipStrm@write[buf, 0, numRead]
			];
			zipStrm@closeEntry[];
			zipStrm@close[];
			fis@close[];
		]
	]


(********************************  Utilities  **********************************)

(* Test as a string to avoid the symbol FrontEndObject forcing loading of System`FEDump, etc. *)
hasFrontEnd[] := ToString[Head[$FrontEnd]] === "FrontEndObject"

hasLocalFrontEnd[] := hasFrontEnd[] && MathLink`CallFrontEnd[FrontEnd`Value["$MachineID"]] === $MachineID


End[]
