(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.1'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    766705,      26119]*)
(*NotebookOutlinePosition[    768628,      26181]*)
(*  CellTagsIndexPosition[    768358,      26168]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Differential-Algebraic Equations", "Chapter"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->{"c:1", "b:10", "ndsg:4.0"}],

Cell["\<\
In general, a system of ordinary differential equations (ODEs) can be \
expressed in the normal form,\
\>", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        SuperscriptBox["x", "\[Prime]",
          MultilineFunction->None], "=", 
        RowBox[{
          StyleBox["f",
            FontWeight->"Bold"], "(", \(t, x\), ")"}]}], 
      TraditionalForm]], "DisplayMath"],

Cell[TextData[{
  "The derivatives of the dependent variables ",
  StyleBox["x",
    FontSlant->"Italic"],
  " are expressed explicitly in terms of the independent variable, ",
  StyleBox["t",
    FontSlant->"Italic"],
  ", and the dependent variables, ",
  StyleBox["x",
    FontSlant->"Italic"],
  ". As long as the function ",
  StyleBox["f",
    FontSlant->"Italic"],
  " has sufficient continuity, a unique solution can always be found for an \
initial value problem where the values of the dependent variables are given \
at a specific value of the independent variable."
}], "Text"],

Cell["\<\
With differential-algebraic equations (DAEs), the derivatives are not, in \
general, expressed explicitly. In fact, derivatives of some of the dependent \
variables typically do not  appear in the equations. The general form of a \
system of DAEs is\
\>", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{"F", "(", 
          RowBox[{"t", ",", "x", ",", 
            SuperscriptBox["x", "\[Prime]",
              MultilineFunction->None]}], ")"}], "=", "0"}], 
      TraditionalForm]], "NumberedEquation"],

Cell[TextData[{
  "where the Jacobian with respect to ",
  StyleBox["x",
    FontSlant->"Italic"],
  "', ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(\[PartialD]F\), "/", 
          RowBox[{"\[PartialD]", 
            SuperscriptBox["x", "\[Prime]",
              MultilineFunction->None]}]}], TraditionalForm]]],
  "may be singular."
}], "Text"],

Cell[TextData[{
  "A system of DAEs can be converted to a system of ODEs by differentiating \
it with respect to the independent variable ",
  StyleBox["t",
    FontSlant->"Italic"],
  ". The ",
  StyleBox["index",
    FontSlant->"Italic"],
  " of a DAE is effectively the number of times you need to differentiate the \
DAEs to get a system of ODEs. Even though the differentiation is possible, it \
is not generally used as a computational technique because properties of the \
original DAEs are often lost in numerical simulations of the differentiated \
equations."
}], "Text"],

Cell[TextData[{
  "Thus, numerical methods for DAEs are designed to work with the general \
form of a system of DAEs. The methods in ",
  StyleBox["NDSolve", "MR"],
  " are designed to generally solve index-1 DAEs, but may work for higher \
index problems as well."
}], "Text"],

Cell["\<\
This documentation will show numerous examples which illustrate some of the \
differences between solving DAEs and ODEs.\
\>", "Text"],

Cell["\<\
This loads packages which will be used in the examples and turns off a \
message.\
\>", "MathCaption"],

Cell[BoxData[{
    \(Needs["\<DifferentialEquations`InterpolatingFunctionAnatomy`\>"]; 
    Needs["\<Graphics`Graphics`\>"];\), "\n", 
    \(\(Off[Graphics::gptn];\)\)}], "Input",
  CellLabel->"In[1]:=",
  InitializationCell->True],

Cell[TextData[{
  "The specification of initial conditions is quite different for DAEs than \
for ODEs. For ODEs, as already mentioned, a set of initial conditions \
uniquely determines a solution. For DAEs, the situation is not nearly so \
simple; it may even be difficult to find initial conditions that satisfy the \
equations at all. To better understand this issue, consider the following \
example [",
  ButtonBox["AP98",
    ButtonData:>{"References.nb", "AP98"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell["\<\
Here is a system of DAEs with three equations, but only one differential \
term.\
\>", "MathCaption"],

Cell[BoxData[
    RowBox[{
      RowBox[{"DAE", "=", 
        RowBox[{"(", GridBox[{
              {
                RowBox[{
                  RowBox[{
                    SuperscriptBox[\(x\_1\), "\[Prime]",
                      MultilineFunction->None], "[", "t", "]"}], 
                  "\[Equal]", \(x\_3[t]\)}]},
              {\(x\_2[t]\ \((1 - x\_2[t])\) \[Equal] 0\)},
              {\(x\_1[t]\ x\_2[t] + x\_3[t]\ \((1 - x\_2[t])\) \[Equal] t\)}
              }], ")"}]}], ";"}]], "Input",
  CellLabel->"In[3]:="],

Cell[TextData[{
  "The initial conditions are clearly not free; the second equation requires \
that ",
  Cell[BoxData[
      \(\(\(\ \)\(x\_2[t\_0]\)\)\)]],
  "is either 0 or 1.",
  " "
}], "Text"],

Cell[TextData[{
  "This solves the system of DAEs starting with a specified initial condition \
for the derivative of ",
  Cell[BoxData[
      \(TraditionalForm\`x\_1\)]],
  ".",
  " "
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol\  = \ 
      NDSolve[{DAE, \(x\_1'\)[0] \[Equal] \ 1}, {x\_1, x\_2, x\_3}, \ {t, 0, 
          1}]\)], "Input",
  CellLabel->"In[4]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(x\_1\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{\(x\_2\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{\(x\_3\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False]}]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[4]="]
}, Open  ]],

Cell[TextData[{
  "To get this solution, ",
  StyleBox["NDSolve", "MR"],
  " first searches for initial conditions that satisfy the equations, using a \
combination of ",
  StyleBox["Solve", "MR"],
  " and a procedure much like ",
  StyleBox["FindRoot", "MR"],
  ". Once consistent initial conditions are found, the DAE is solved using \
the IDA method."
}], "Text"],

Cell[TextData[{
  "This shows the initial conditions found by ",
  StyleBox["NDSolve", "MR"],
  ". "
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \({{\(x\_1'\)[0]}, \ {x\_1[0], x\_2[0], x\_3[0]}}\  /. \ 
      First[sol]\)], "Input",
  CellLabel->"In[5]:="],

Cell[BoxData[
    \({{1.`}, {0.`, 1.`, 1.`}}\)], "Output",
  CellLabel->"Out[5]="]
}, Open  ]],

Cell[TextData[{
  "This shows a plot of the solution. The solution ",
  Cell[BoxData[
      \(\(\(x\_2[0]\)\(\ \)\)\)]],
  "is obscured by the solution ",
  Cell[BoxData[
      \(\(\(\ \)\(x\_3[0]\)\)\)]],
  ", which has the same constant value of 1."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot[
      Evaluate[{x\_1[t], x\_2[t], x\_3[t]}\  /. \ First[sol]], {t, 0, 
        1}, \ \[IndentingNewLine]PlotStyle \[Rule] {{RGBColor[1, 0, 
              0]}, \ {RGBColor[0, 0, 0]}, \ {RGBColor[0, 0, 
              1]}}]; \)], "Input",
  CellLabel->"In[6]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.0147151 0.588604 [
[.21429 .00222 -9 -9 ]
[.21429 .00222 9 0 ]
[.40476 .00222 -9 -9 ]
[.40476 .00222 9 0 ]
[.59524 .00222 -9 -9 ]
[.59524 .00222 9 0 ]
[.78571 .00222 -9 -9 ]
[.78571 .00222 9 0 ]
[.97619 .00222 -3 -9 ]
[.97619 .00222 3 0 ]
[.01131 .13244 -18 -4.5 ]
[.01131 .13244 0 4.5 ]
[.01131 .25016 -18 -4.5 ]
[.01131 .25016 0 4.5 ]
[.01131 .36788 -18 -4.5 ]
[.01131 .36788 0 4.5 ]
[.01131 .4856 -18 -4.5 ]
[.01131 .4856 0 4.5 ]
[.01131 .60332 -6 -4.5 ]
[.01131 .60332 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .01472 m
.21429 .02097 L
s
[(0.2)] .21429 .00222 0 1 Mshowa
.40476 .01472 m
.40476 .02097 L
s
[(0.4)] .40476 .00222 0 1 Mshowa
.59524 .01472 m
.59524 .02097 L
s
[(0.6)] .59524 .00222 0 1 Mshowa
.78571 .01472 m
.78571 .02097 L
s
[(0.8)] .78571 .00222 0 1 Mshowa
.97619 .01472 m
.97619 .02097 L
s
[(1)] .97619 .00222 0 1 Mshowa
.125 Mabswid
.07143 .01472 m
.07143 .01847 L
s
.11905 .01472 m
.11905 .01847 L
s
.16667 .01472 m
.16667 .01847 L
s
.2619 .01472 m
.2619 .01847 L
s
.30952 .01472 m
.30952 .01847 L
s
.35714 .01472 m
.35714 .01847 L
s
.45238 .01472 m
.45238 .01847 L
s
.5 .01472 m
.5 .01847 L
s
.54762 .01472 m
.54762 .01847 L
s
.64286 .01472 m
.64286 .01847 L
s
.69048 .01472 m
.69048 .01847 L
s
.7381 .01472 m
.7381 .01847 L
s
.83333 .01472 m
.83333 .01847 L
s
.88095 .01472 m
.88095 .01847 L
s
.92857 .01472 m
.92857 .01847 L
s
.25 Mabswid
0 .01472 m
1 .01472 L
s
.02381 .13244 m
.03006 .13244 L
s
[(0.2)] .01131 .13244 1 0 Mshowa
.02381 .25016 m
.03006 .25016 L
s
[(0.4)] .01131 .25016 1 0 Mshowa
.02381 .36788 m
.03006 .36788 L
s
[(0.6)] .01131 .36788 1 0 Mshowa
.02381 .4856 m
.03006 .4856 L
s
[(0.8)] .01131 .4856 1 0 Mshowa
.02381 .60332 m
.03006 .60332 L
s
[(1)] .01131 .60332 1 0 Mshowa
.125 Mabswid
.02381 .04415 m
.02756 .04415 L
s
.02381 .07358 m
.02756 .07358 L
s
.02381 .10301 m
.02756 .10301 L
s
.02381 .16187 m
.02756 .16187 L
s
.02381 .1913 m
.02756 .1913 L
s
.02381 .22073 m
.02756 .22073 L
s
.02381 .27959 m
.02756 .27959 L
s
.02381 .30902 m
.02756 .30902 L
s
.02381 .33845 m
.02756 .33845 L
s
.02381 .39731 m
.02756 .39731 L
s
.02381 .42674 m
.02756 .42674 L
s
.02381 .45617 m
.02756 .45617 L
s
.02381 .51503 m
.02756 .51503 L
s
.02381 .54446 m
.02756 .54446 L
s
.02381 .57389 m
.02756 .57389 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.5 Mabswid
.02381 .01472 m
.06244 .03859 L
.10458 .06463 L
.14415 .08909 L
.18221 .11261 L
.22272 .13765 L
.26171 .16175 L
.30316 .18736 L
.34309 .21204 L
.3815 .23578 L
.42237 .26104 L
.46172 .28536 L
.49955 .30874 L
.53984 .33364 L
.57861 .3576 L
.61984 .38308 L
.65954 .40762 L
.69774 .43123 L
.73838 .45635 L
.77751 .48053 L
.81909 .50623 L
.85916 .53099 L
.89771 .55482 L
.93871 .58016 L
.97619 .60332 L
s
0 0 0 r
.02381 .60332 m
.06244 .60332 L
.10458 .60332 L
.14415 .60332 L
.18221 .60332 L
.22272 .60332 L
.26171 .60332 L
.30316 .60332 L
.34309 .60332 L
.3815 .60332 L
.42237 .60332 L
.46172 .60332 L
.49955 .60332 L
.53984 .60332 L
.57861 .60332 L
.61984 .60332 L
.65954 .60332 L
.69774 .60332 L
.73838 .60332 L
.77751 .60332 L
.81909 .60332 L
.85916 .60332 L
.89771 .60332 L
.93871 .60332 L
.97619 .60332 L
s
0 0 1 r
.02381 .60332 m
.06244 .60332 L
.10458 .60332 L
.14415 .60332 L
.18221 .60332 L
.22272 .60332 L
.26171 .60332 L
.30316 .60332 L
.34309 .60332 L
.3815 .60332 L
.42237 .60332 L
.46172 .60332 L
.49955 .60332 L
.53984 .60332 L
.57861 .60332 L
.61984 .60332 L
.65954 .60332 L
.69774 .60332 L
.73838 .60332 L
.77751 .60332 L
.81909 .60332 L
.85916 .60332 L
.89771 .60332 L
.93871 .60332 L
.97619 .60332 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[6]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00@P3oool2000000@0oooo0P0000030?ooo`@000009@3oool2000000@0oooo0P0000050?oo
o`<000008`3oool2000000@0oooo0P0000030?ooo`<000009P3oool2000000@0oooo0P0000040?oo
o`800000:`3oool5000000D0oooo00110?ooo`040000003oool0oooo000000P0oooo00<000000?oo
o`3oool09@3oool010000000oooo0?ooo`00000;0?ooo`030000003oool0oooo0240oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`00000T0?ooo`040000003oool0oooo000000P0
oooo00@000000?ooo`3oool00000;03oool00`000000oooo0?ooo`050?ooo`00@@3oool010000000
oooo0?ooo`0000090?ooo`030000003oool0oooo02@0oooo00@000000?ooo`3oool00000203oool5
00000280oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`00000T0?ooo`040000
003oool0oooo000000P0oooo00@000000?ooo`3oool00000;03oool00`000000oooo0?ooo`050?oo
o`00@@3oool010000000oooo0?ooo`00000:0?ooo`030000003oool0oooo02<0oooo00@000000?oo
o`3oool00000203oool010000000oooo0?ooo`00000S0?ooo`040000003oool0oooo000000P0oooo
0`00000U0?ooo`040000003oool0oooo000000T0oooo0P00000]0?ooo`030000003oool0oooo00D0
oooo00110?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool00000903oool01000
0000oooo0?ooo`0000090?ooo`030000003oool0000002<0oooo00@000000?ooo`3oool000002@3o
ool00`000000oooo0?ooo`0T0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
0000;03oool00`000000oooo0?ooo`050?ooo`00@P3oool2000000X0oooo0P00000V0?ooo`800000
2`3oool2000002@0oooo0P00000:0?ooo`<000009@3oool2000000X0oooo0P00000/0?ooo`800000
1`3oool00?l0oooo8@3oool001D0oooo00<000000?ooo`3oool0o`3oool90?ooo`005@3oool00`00
0000oooo0?ooo`3o0?ooo`T0oooo000E0?ooo`030000003oool0oooo0?l0oooo2@3oool001D0oooo
00<000000?ooo`3oool0o`3oool90?ooo`003`3oool6000000030?l00000000000000?l000002000
00010?ooo`005@3oool00`000000o`000?l0000:0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool01@3oool001D0oooo00<000000?ooo`3oool00P3o000_0?ooo`030000003oool0oooo
0340oooo00<000000?ooo`3oool0<03oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo
0340oooo00<000000?ooo`3oool01@3oool001D0oooo00<000000?ooo`3oool00P3oool00`3o0000
oooo0?ooo`3o0?ooo`@0oooo000E0?ooo`030000003oool0oooo00<0oooo0P3o003o0?ooo`@0oooo
000E0?ooo`030000003oool0oooo00D0oooo0P3o003o0?ooo`80oooo000E0?ooo`030000003oool0
oooo00L0oooo00<0o`000?ooo`3oool0oP3oool001D0oooo00<000000?ooo`3oool0203oool20?l0
0?h0oooo000E0?ooo`8000002`3oool00`3o0000oooo0?ooo`3k0?ooo`005@3oool00`000000oooo
0?ooo`0;0?ooo`80o`00n`3oool001D0oooo00<000000?ooo`3oool03@3oool00`3o0000oooo0?oo
o`3h0?ooo`005@3oool00`000000oooo0?ooo`0>0?ooo`80o`00n03oool001D0oooo00<000000?oo
o`3oool0403oool00`3o0000oooo0?ooo`3e0?ooo`005@3oool00`000000oooo0?ooo`0A0?ooo`80
o`00m@3oool001D0oooo00<000000?ooo`3oool04`3oool00`3o0000oooo0?ooo`3b0?ooo`005@3o
ool00`000000oooo0?ooo`0D0?ooo`80o`00lP3oool001D0oooo0P00000G0?ooo`80o`00l03oool0
01D0oooo00<000000?ooo`3oool0603oool20?l00>h0oooo000E0?ooo`030000003oool0oooo01X0
oooo0P3o003/0?ooo`005@3oool00`000000oooo0?ooo`0L0?ooo`80o`00jP3oool001D0oooo00<0
00000?ooo`3oool07P3oool00`3o0000oooo0?ooo`3W0?ooo`005@3oool00`000000oooo0?ooo`0O
0?ooo`80o`00i`3oool001D0oooo00<000000?ooo`3oool08@3oool00`3o0000oooo0?ooo`3T0?oo
o`005@3oool00`000000oooo0?ooo`0R0?ooo`030?l0003oool0oooo0><0oooo000E0?ooo`800000
903oool20?l00><0oooo000E0?ooo`030000003oool0oooo02D0oooo00<0o`000?ooo`3oool0h03o
ool001D0oooo00<000000?ooo`3oool09P3oool20?l00>00oooo000E0?ooo`030000003oool0oooo
02P0oooo00<0o`000?ooo`3oool0g@3oool001D0oooo00<000000?ooo`3oool0:@3oool20?l00=d0
oooo000E0?ooo`030000003oool0oooo02/0oooo0P3o003K0?ooo`000P3oool2000000@0oooo0P00
00030?ooo`@00000103oool00`000000oooo0?ooo`0]0?ooo`80o`00f@3oool000050?ooo`000000
oooo0?ooo`000000203oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo02l0oooo0P3o
003G0?ooo`0000D0oooo0000003oool0oooo000000090?ooo`030000003oool0oooo00@0oooo0`00
000a0?ooo`80o`00e@3oool000050?ooo`000000oooo0?ooo`0000002P3oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo03<0oooo00<0o`000?ooo`3oool0dP3oool000050?ooo`000000
oooo0?ooo`000000203oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo03@0oooo
0P3o003B0?ooo`000P3oool2000000X0oooo0P0000050?ooo`030000003oool0oooo03H0oooo00<0
o`000?ooo`3oool0c`3oool001D0oooo00<000000?ooo`3oool0=`3oool20?l00<l0oooo000E0?oo
o`030000003oool0oooo03T0oooo0P3o003=0?ooo`005@3oool00`000000oooo0?ooo`0k0?ooo`03
0?l0003oool0oooo0<X0oooo000E0?ooo`030000003oool0oooo03`0oooo0P3o003:0?ooo`005@3o
ool2000003l0oooo00<0o`000?ooo`3oool0a`3oool001D0oooo00<000000?ooo`3oool0?`3oool2
0?l00<L0oooo000E0?ooo`030000003oool0oooo0440oooo00<0o`000?ooo`3oool0a03oool001D0
oooo00<000000?ooo`3oool0@P3oool20?l00<@0oooo000E0?ooo`030000003oool0oooo04@0oooo
0P3o00320?ooo`005@3oool00`000000oooo0?ooo`160?ooo`030?l0003oool0oooo0;l0oooo000E
0?ooo`030000003oool0oooo04L0oooo0P3o002o0?ooo`005@3oool00`000000oooo0?ooo`190?oo
o`030?l0003oool0oooo0;`0oooo000E0?ooo`800000B`3oool20?l00;`0oooo000E0?ooo`030000
003oool0oooo04`0oooo00<0o`000?ooo`3oool0^@3oool001D0oooo00<000000?ooo`3oool0C@3o
ool20?l00;T0oooo000E0?ooo`030000003oool0oooo04l0oooo0P3o002g0?ooo`005@3oool00`00
0000oooo0?ooo`1A0?ooo`030?l0003oool0oooo0;@0oooo000E0?ooo`030000003oool0oooo0580
oooo0P3o002d0?ooo`005@3oool00`000000oooo0?ooo`1D0?ooo`030?l0003oool0oooo0;40oooo
000E0?ooo`030000003oool0oooo05D0oooo0P3o002a0?ooo`005@3oool2000005P0oooo0P3o002_
0?ooo`005@3oool00`000000oooo0?ooo`1I0?ooo`030?l0003oool0oooo0:`0oooo000E0?ooo`03
0000003oool0oooo05X0oooo0P3o002/0?ooo`005@3oool00`000000oooo0?ooo`1L0?ooo`80o`00
ZP3oool001D0oooo00<000000?ooo`3oool0GP3oool00`3o0000oooo0?ooo`2W0?ooo`005@3oool0
0`000000oooo0?ooo`1O0?ooo`80o`00Y`3oool00080oooo0P0000040?ooo`8000001@3oool30000
00<0oooo00<000000?ooo`3oool0H@3oool00`3o0000oooo0?ooo`2T0?ooo`0000D0oooo0000003o
ool0oooo0000000;0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0HP3oool20?l0
0:@0oooo00001@3oool000000?ooo`3oool0000000P0oooo1@0000030?ooo`<00000I03oool20?l0
0:80oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool00000103oool0
0`000000oooo0?ooo`1V0?ooo`030?l0003oool0oooo09l0oooo00001@3oool000000?ooo`3oool0
000000T0oooo00<000000?ooo`000000103oool00`000000oooo0?ooo`1W0?ooo`80o`00W`3oool0
0080oooo0P00000;0?ooo`800000103oool00`000000oooo0?ooo`1Y0?ooo`030?l0003oool0oooo
09`0oooo000E0?ooo`030000003oool0oooo06X0oooo0P3o002L0?ooo`005@3oool00`000000oooo
0?ooo`1/0?ooo`030?l0003oool0oooo09T0oooo000E0?ooo`030000003oool0oooo06d0oooo0P3o
002I0?ooo`005@3oool00`000000oooo0?ooo`1_0?ooo`80o`00U`3oool001D0oooo0P00001b0?oo
o`030?l0003oool0oooo09@0oooo000E0?ooo`030000003oool0oooo0780oooo0P3o002D0?ooo`00
5@3oool00`000000oooo0?ooo`1d0?ooo`030?l0003oool0oooo0940oooo000E0?ooo`030000003o
ool0oooo07D0oooo0P3o002A0?ooo`005@3oool00`000000oooo0?ooo`1g0?ooo`80o`00S`3oool0
01D0oooo00<000000?ooo`3oool0N@3oool00`3o0000oooo0?ooo`2<0?ooo`005@3oool00`000000
oooo0?ooo`1j0?ooo`80o`00S03oool001D0oooo00<000000?ooo`3oool0O03oool20?l008X0oooo
000E0?ooo`800000O`3oool00`3o0000oooo0?ooo`270?ooo`005@3oool00`000000oooo0?ooo`1o
0?ooo`80o`00Q`3oool001D0oooo00<000000?ooo`3oool0P@3oool00`3o0000oooo0?ooo`240?oo
o`005@3oool00`000000oooo0?ooo`220?ooo`80o`00Q03oool001D0oooo00<000000?ooo`3oool0
Q03oool20?l00880oooo000E0?ooo`030000003oool0oooo08H0oooo00<0o`000?ooo`3oool0O`3o
ool001D0oooo00<000000?ooo`3oool0Q`3oool20?l007l0oooo000E0?ooo`030000003oool0oooo
08T0oooo00<0o`000?ooo`3oool0O03oool001D0oooo0P00002;0?ooo`80o`00O03oool001D0oooo
00<000000?ooo`3oool0S03oool20?l007X0oooo000E0?ooo`030000003oool0oooo08h0oooo0P3o
001h0?ooo`005@3oool00`000000oooo0?ooo`2@0?ooo`80o`00MP3oool001D0oooo00<000000?oo
o`3oool0TP3oool20?l007@0oooo000E0?ooo`030000003oool0oooo09@0oooo00<0o`000?ooo`3o
ool0L@3oool00080oooo0P0000040?ooo`8000000`3oool3000000D0oooo00<000000?ooo`3oool0
U@3oool20?l00740oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool0
0000103oool00`000000oooo0?ooo`2G0?ooo`030?l0003oool0oooo06h0oooo00001@3oool00000
0?ooo`3oool0000000P0oooo00@000000?ooo`3oool00000103oool3000009P0oooo0P3o001^0?oo
o`0000D0oooo0000003oool0oooo000000080?ooo`<000001@3oool00`000000oooo0?ooo`2J0?oo
o`80o`00K03oool000050?ooo`000000oooo0?ooo`0000002@3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo09`0oooo00<0o`000?ooo`3oool0J@3oool00080oooo0P00000:0?ooo`<0
0000103oool00`000000oooo0?ooo`2M0?ooo`80o`00J@3oool001D0oooo00<000000?ooo`3oool0
W`3oool00`3o0000oooo0?ooo`1V0?ooo`005@3oool00`000000oooo0?ooo`2P0?ooo`80o`00IP3o
ool001D0oooo00<000000?ooo`3oool0XP3oool00`3o0000oooo0?ooo`1S0?ooo`005@3oool00`00
0000oooo0?ooo`2S0?ooo`80o`00H`3oool001D0oooo0P00002V0?ooo`80o`00H@3oool001D0oooo
00<000000?ooo`3oool0Y`3oool00`3o0000oooo0?ooo`1N0?ooo`005@3oool00`000000oooo0?oo
o`2X0?ooo`80o`00GP3oool001D0oooo00<000000?ooo`3oool0ZP3oool00`3o0000oooo0?ooo`1K
0?ooo`005@3oool00`000000oooo0?ooo`2[0?ooo`80o`00F`3oool001D0oooo00<000000?ooo`3o
ool0[@3oool20?l005T0oooo000E0?ooo`030000003oool0oooo0:l0oooo00<0o`000?ooo`3oool0
EP3oool001D0oooo00<000000?ooo`3oool0/03oool20?l005H0oooo000E0?ooo`800000/`3oool2
0?l005@0oooo000E0?ooo`030000003oool0oooo0;@0oooo00<0o`000?ooo`3oool0D@3oool001D0
oooo00<000000?ooo`3oool0]@3oool20?l00540oooo000E0?ooo`030000003oool0oooo0;L0oooo
00<0o`000?ooo`3oool0CP3oool001D0oooo00<000000?ooo`3oool0^03oool20?l004h0oooo000E
0?ooo`030000003oool0oooo0;X0oooo0P3o001<0?ooo`005@3oool00`000000oooo0?ooo`2l0?oo
o`030?l0003oool0oooo04T0oooo000E0?ooo`030000003oool0oooo0;d0oooo0P3o00190?ooo`00
5@3oool200000<00oooo00<0o`000?ooo`3oool0AP3oool001D0oooo00<000000?ooo`3oool0`03o
ool20?l004H0oooo000E0?ooo`030000003oool0oooo0<80oooo00<0o`000?ooo`3oool0@`3oool0
01D0oooo00<000000?ooo`3oool0``3oool20?l004<0oooo000E0?ooo`030000003oool0oooo0<D0
oooo0P3o00110?ooo`005@3oool00`000000oooo0?ooo`370?ooo`030?l0003oool0oooo03h0oooo
00020?ooo`800000103oool2000000@0oooo0P0000050?ooo`030000003oool0oooo0<P0oooo0P3o
000n0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo000000@0oooo
00<000000?ooo`3oool0bP3oool00`3o0000oooo0?ooo`0k0?ooo`0000D0oooo0000003oool0oooo
000000080?ooo`040000003oool0oooo000000@0oooo0`00003;0?ooo`80o`00>`3oool000050?oo
o`000000oooo0?ooo`0000002@3oool2000000D0oooo00<000000?ooo`3oool0c@3oool00`3o0000
oooo0?ooo`0h0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo0000
00@0oooo00<000000?ooo`3oool0cP3oool20?l003P0oooo00020?ooo`8000002P3oool2000000D0
oooo00<000000?ooo`3oool0d03oool20?l003H0oooo000E0?ooo`030000003oool0oooo0=80oooo
00<0o`000?ooo`3oool0<`3oool001D0oooo00<000000?ooo`3oool0d`3oool20?l003<0oooo000E
0?ooo`030000003oool0oooo0=D0oooo00<0o`000?ooo`3oool0<03oool001D0oooo00<000000?oo
o`3oool0eP3oool20?l00300oooo000E0?ooo`800000f@3oool20?l002h0oooo000E0?ooo`030000
003oool0oooo0=X0oooo0P3o000/0?ooo`005@3oool00`000000oooo0?ooo`3L0?ooo`80o`00:P3o
ool001D0oooo00<000000?ooo`3oool0gP3oool20?l002P0oooo000E0?ooo`030000003oool0oooo
0>00oooo00<0o`000?ooo`3oool09@3oool001D0oooo00<000000?ooo`3oool0h@3oool20?l002D0
oooo000E0?ooo`030000003oool0oooo0><0oooo00<0o`000?ooo`3oool08P3oool001D0oooo00<0
00000?ooo`3oool0i03oool20?l00280oooo000E0?ooo`800000i`3oool20?l00200oooo000E0?oo
o`030000003oool0oooo0>P0oooo00<0o`000?ooo`3oool07@3oool001D0oooo00<000000?ooo`3o
ool0j@3oool20?l001d0oooo000E0?ooo`030000003oool0oooo0>/0oooo00<0o`000?ooo`3oool0
6P3oool001D0oooo00<000000?ooo`3oool0k03oool20?l001X0oooo000E0?ooo`030000003oool0
oooo0>h0oooo00<0o`000?ooo`3oool05`3oool001D0oooo00<000000?ooo`3oool0k`3oool20?l0
01L0oooo000E0?ooo`030000003oool0oooo0?40oooo0P3o000E0?ooo`005@3oool200000?@0oooo
00<0o`000?ooo`3oool04P3oool001D0oooo00<000000?ooo`3oool0m03oool20?l00180oooo000E
0?ooo`030000003oool0oooo0?H0oooo00<0o`000?ooo`3oool03`3oool001D0oooo00<000000?oo
o`3oool0m`3oool20?l000l0oooo000E0?ooo`030000003oool0oooo0?T0oooo0P3o000=0?ooo`00
5@3oool00`000000oooo0?ooo`3k0?ooo`030?l0003oool0oooo00X0oooo000=0?ooo`D000000`3o
ool00`000000oooo0?ooo`3l0?ooo`80o`002P3oool000l0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`3n0?ooo`80o`00203oool000l0oooo00<000000?ooo`3oool00`3ooooo0000
o`D0003o1`3oool000l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`3o0?ooo`T0
oooo000?0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0o`3oool90?ooo`003P3o
ool2000000D0oooo00<000000?ooo`3oool0o`3oool90?ooo`005@3oool00`000000oooo0?ooo`3o
0?ooo`T0oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0829852, -0.07747, \
0.00386179, 0.00624851}}]
}, Open  ]],

Cell["\<\
However, there may not be a solution from all initial conditions that satisfy \
the equations. \
\>", "Text"],

Cell[TextData[{
  "This tries to find a solution with ",
  Cell[BoxData[
      \(x\_2[0]\)]],
  "starting from steady state with derivative 0.",
  " "
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sols\  = \ 
      NDSolve[{DAE, \(x\_1'\)[0]\  \[Equal] \ 0}, {x\_1, x\_2, x\_3}, \ {t, 
          0, 1}]\)], "Input",
  CellLabel->"In[7]:="],

Cell[BoxData[
    \(NDSolve::"nderr" \(\(:\)\(\ \)\) 
      "Error test failure at \!\(t\) == \!\(0.`\); unable to continue."\)], \
"Message",
  CellLabel->"From In[7]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(x\_1\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 0.`}}, "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{\(x\_2\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 0.`}}, "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{\(x\_3\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 0.`}}, "<>"]\),
              False,
              Editable->False]}]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[7]="]
}, Open  ]],

Cell[TextData[{
  "This shows the initial conditions found by ",
  StyleBox["NDSolve", "MR"],
  ". "
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \({{\(x\_1'\)[0]}, \ {x\_1[0], x\_2[0], x\_3[0]}}\  /. \ 
      First[sols]\)], "Input",
  CellLabel->"In[8]:="],

Cell[BoxData[
    \({{0.`}, {0.`, 1.`, 0.`}}\)], "Output",
  CellLabel->"Out[8]="]
}, Open  ]],

Cell[TextData[{
  "If you look at the equations with ",
  Cell[BoxData[
      \(x\_2\)]],
  "set to 1, you can see why it is not possible to advance beyond ",
  StyleBox["t",
    FontSlant->"Italic"],
  " = 1."
}], "Text"],

Cell[TextData[{
  "Substitute ",
  Cell[BoxData[
      \(x\_2[t] = \ 1\)]],
  " into the equations."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DAE\  /. \ x\_2[t] \[Rule] 1\)], "Input",
  CellLabel->"In[9]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              SuperscriptBox[\(x\_1\), "\[Prime]",
                MultilineFunction->None], "[", "t", "]"}], 
            "\[Equal]", \(x\_3[t]\)}], "}"}], ",", \({True}\), 
        ",", \({x\_1[t] \[Equal] t}\)}], "}"}]], "Output",
  CellLabel->"Out[9]="]
}, Open  ]],

Cell[TextData[{
  "The middle equation effectively drops out. If you differentiate the last \
equation with",
  Cell[BoxData[
      \(\(\(\ \)\(x\_2[t] = \ 1\)\)\)]],
  ", you get the condition ",
  Cell[BoxData[
      \(\(x\_1'\)[t] \[Equal] 1\)]],
  ", but then the first equation is inconsistent with the value of ",
  Cell[BoxData[
      \(x\_3[t]\  = \ 0\)]],
  "in the initial conditions. "
}], "Text"],

Cell[TextData[{
  "It turns out that the only solution with ",
  Cell[BoxData[
      \(x\_2[t] = \ 1\)]],
  "is {",
  Cell[BoxData[
      \(x\_2[t] = \ t\)]],
  ", ",
  Cell[BoxData[
      \(\(\(x\_2[t] = \ 1\)\(,\)\)\)]],
  Cell[BoxData[
      \(x\_3[t] = \ 1\)]],
  "}, and along this solution, the system has index 2.",
  " "
}], "Text"],

Cell[TextData[{
  "The other set of solutions for the problem is when ",
  Cell[BoxData[
      \(x\_2[t] = \ 0\)]],
  ".",
  " ",
  "You can find these by specifying that as an initial condition."
}], "Text"],

Cell[TextData[{
  "This finds a solution with ",
  Cell[BoxData[
      \(x\_2[t] = \ 0\)]],
  ". It is also necessary to specify a value for ",
  Cell[BoxData[
      \(x\_1[0]\)]],
  " because it is a differential variable."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol0\  = \ 
      NDSolve[{DAE, x\_1[0]\  \[Equal] \ 1, \ x\_2[0]\  \[Equal] \ 0}, {x\_1, 
          x\_2, x\_3}, \ {t, 0, 1}]\)], "Input",
  CellLabel->"In[10]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(x\_1\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{\(x\_2\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{\(x\_3\), "\[Rule]", 
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False]}]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[10]="]
}, Open  ]],

Cell["\<\
This shows a plot of the nonzero components of the solution. \
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot[
      Evaluate[{x\_1[t], x\_3[t]}\  /. \ First[sol0]], {t, 0, 
        1}, \ \[IndentingNewLine]PlotStyle \[Rule] {{RGBColor[1, 0, 
              0]}, \ {RGBColor[0, 0, 1]}}]; \)], "Input",
  CellLabel->"In[11]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.0147151 0.392403 [
[.21429 .00222 -9 -9 ]
[.21429 .00222 9 0 ]
[.40476 .00222 -9 -9 ]
[.40476 .00222 9 0 ]
[.59524 .00222 -9 -9 ]
[.59524 .00222 9 0 ]
[.78571 .00222 -9 -9 ]
[.78571 .00222 9 0 ]
[.97619 .00222 -3 -9 ]
[.97619 .00222 3 0 ]
[.01131 .0932 -18 -4.5 ]
[.01131 .0932 0 4.5 ]
[.01131 .17168 -18 -4.5 ]
[.01131 .17168 0 4.5 ]
[.01131 .25016 -18 -4.5 ]
[.01131 .25016 0 4.5 ]
[.01131 .32864 -18 -4.5 ]
[.01131 .32864 0 4.5 ]
[.01131 .40712 -6 -4.5 ]
[.01131 .40712 0 4.5 ]
[.01131 .4856 -18 -4.5 ]
[.01131 .4856 0 4.5 ]
[.01131 .56408 -18 -4.5 ]
[.01131 .56408 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .01472 m
.21429 .02097 L
s
[(0.2)] .21429 .00222 0 1 Mshowa
.40476 .01472 m
.40476 .02097 L
s
[(0.4)] .40476 .00222 0 1 Mshowa
.59524 .01472 m
.59524 .02097 L
s
[(0.6)] .59524 .00222 0 1 Mshowa
.78571 .01472 m
.78571 .02097 L
s
[(0.8)] .78571 .00222 0 1 Mshowa
.97619 .01472 m
.97619 .02097 L
s
[(1)] .97619 .00222 0 1 Mshowa
.125 Mabswid
.07143 .01472 m
.07143 .01847 L
s
.11905 .01472 m
.11905 .01847 L
s
.16667 .01472 m
.16667 .01847 L
s
.2619 .01472 m
.2619 .01847 L
s
.30952 .01472 m
.30952 .01847 L
s
.35714 .01472 m
.35714 .01847 L
s
.45238 .01472 m
.45238 .01847 L
s
.5 .01472 m
.5 .01847 L
s
.54762 .01472 m
.54762 .01847 L
s
.64286 .01472 m
.64286 .01847 L
s
.69048 .01472 m
.69048 .01847 L
s
.7381 .01472 m
.7381 .01847 L
s
.83333 .01472 m
.83333 .01847 L
s
.88095 .01472 m
.88095 .01847 L
s
.92857 .01472 m
.92857 .01847 L
s
.25 Mabswid
0 .01472 m
1 .01472 L
s
.02381 .0932 m
.03006 .0932 L
s
[(0.2)] .01131 .0932 1 0 Mshowa
.02381 .17168 m
.03006 .17168 L
s
[(0.4)] .01131 .17168 1 0 Mshowa
.02381 .25016 m
.03006 .25016 L
s
[(0.6)] .01131 .25016 1 0 Mshowa
.02381 .32864 m
.03006 .32864 L
s
[(0.8)] .01131 .32864 1 0 Mshowa
.02381 .40712 m
.03006 .40712 L
s
[(1)] .01131 .40712 1 0 Mshowa
.02381 .4856 m
.03006 .4856 L
s
[(1.2)] .01131 .4856 1 0 Mshowa
.02381 .56408 m
.03006 .56408 L
s
[(1.4)] .01131 .56408 1 0 Mshowa
.125 Mabswid
.02381 .03434 m
.02756 .03434 L
s
.02381 .05396 m
.02756 .05396 L
s
.02381 .07358 m
.02756 .07358 L
s
.02381 .11282 m
.02756 .11282 L
s
.02381 .13244 m
.02756 .13244 L
s
.02381 .15206 m
.02756 .15206 L
s
.02381 .1913 m
.02756 .1913 L
s
.02381 .21092 m
.02756 .21092 L
s
.02381 .23054 m
.02756 .23054 L
s
.02381 .26978 m
.02756 .26978 L
s
.02381 .2894 m
.02756 .2894 L
s
.02381 .30902 m
.02756 .30902 L
s
.02381 .34826 m
.02756 .34826 L
s
.02381 .36788 m
.02756 .36788 L
s
.02381 .3875 m
.02756 .3875 L
s
.02381 .42674 m
.02756 .42674 L
s
.02381 .44636 m
.02756 .44636 L
s
.02381 .46598 m
.02756 .46598 L
s
.02381 .50522 m
.02756 .50522 L
s
.02381 .52484 m
.02756 .52484 L
s
.02381 .54446 m
.02756 .54446 L
s
.02381 .5837 m
.02756 .5837 L
s
.02381 .60332 m
.02756 .60332 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.5 Mabswid
.02381 .40712 m
.02499 .40712 L
.02605 .40712 L
.02729 .40712 L
.02846 .40712 L
.03053 .40713 L
.03279 .40714 L
.03527 .40715 L
.0379 .40716 L
.04262 .40719 L
.04749 .40724 L
.05205 .40729 L
.06244 .40744 L
.07305 .40764 L
.08274 .40787 L
.10458 .40853 L
.12357 .40927 L
.14429 .41026 L
.18493 .41273 L
.22406 .41579 L
.26565 .41977 L
.30571 .42431 L
.34426 .42933 L
.38527 .43538 L
.42475 .44189 L
.46273 .44879 L
.50315 .45682 L
.54206 .46522 L
.58342 .47486 L
.62326 .48485 L
.66159 .49511 L
.70238 .50672 L
.74164 .51858 L
.77939 .53061 L
.8196 .5441 L
.85828 .55775 L
.89942 .57296 L
.93905 .58831 L
.97619 .60332 L
s
0 0 1 r
.02381 .01472 m
.06244 .03063 L
.10458 .04799 L
.14415 .0643 L
.18221 .07998 L
.22272 .09667 L
.26171 .11274 L
.30316 .12981 L
.34309 .14626 L
.3815 .16209 L
.42237 .17893 L
.46172 .19514 L
.49955 .21073 L
.53984 .22733 L
.57861 .24331 L
.61984 .26029 L
.65954 .27665 L
.69774 .29239 L
.73838 .30914 L
.77751 .32526 L
.81909 .34239 L
.85916 .3589 L
.89771 .37478 L
.93871 .39168 L
.97619 .40712 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[11]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00@P3oool2000000@0oooo0P0000030?ooo`@000009@3oool2000000@0oooo0P0000050?oo
o`<000008`3oool2000000@0oooo0P0000030?ooo`<000009P3oool2000000@0oooo0P0000040?oo
o`800000:`3oool5000000D0oooo00110?ooo`040000003oool0oooo000000P0oooo00<000000?oo
o`3oool09@3oool010000000oooo0?ooo`00000;0?ooo`030000003oool0oooo0240oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`00000T0?ooo`040000003oool0oooo000000P0
oooo00@000000?ooo`3oool00000;03oool00`000000oooo0?ooo`050?ooo`00@@3oool010000000
oooo0?ooo`0000090?ooo`030000003oool0oooo02@0oooo00@000000?ooo`3oool00000203oool5
00000280oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`00000T0?ooo`040000
003oool0oooo000000P0oooo00@000000?ooo`3oool00000;03oool00`000000oooo0?ooo`050?oo
o`00@@3oool010000000oooo0?ooo`00000:0?ooo`030000003oool0oooo02<0oooo00@000000?oo
o`3oool00000203oool010000000oooo0?ooo`00000S0?ooo`040000003oool0oooo000000P0oooo
0`00000U0?ooo`040000003oool0oooo000000T0oooo0P00000]0?ooo`030000003oool0oooo00D0
oooo00110?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool00000903oool01000
0000oooo0?ooo`0000090?ooo`030000003oool0000002<0oooo00@000000?ooo`3oool000002@3o
ool00`000000oooo0?ooo`0T0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
0000;03oool00`000000oooo0?ooo`050?ooo`00@P3oool2000000X0oooo0P00000V0?ooo`800000
2`3oool2000002@0oooo0P00000:0?ooo`<000009@3oool2000000X0oooo0P00000/0?ooo`800000
1`3oool00?l0oooo8@3oool001D0oooo00<000000?ooo`3oool0o`3oool90?ooo`005@3oool00`00
0000oooo0?ooo`3o0?ooo`T0oooo000E0?ooo`030000003oool0oooo0?l0oooo2@3oool001D0oooo
00<000000?ooo`3oool0o`3oool90?ooo`003P3oool700000080003oo`00000:000000005@3oool0
10000000oooo0000o`000?l90?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool01@3o
ool001D0oooo00@000000?ooo`3oool0oooo0`000?l]0?ooo`030000003oool0oooo0340oooo00<0
00000?ooo`3oool0<03oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo0340oooo00<0
00000?ooo`3oool01@3oool001D0oooo00<000000?ooo`3oool0103oool20000ool0oooo0`3oool0
01D0oooo00<000000?ooo`3oool01P3oool30000ool0oooo000E0?ooo`8000002P3oool20000ood0
oooo000E0?ooo`030000003oool0oooo00/0oooo0P000?ok0?ooo`005@3oool00`000000oooo0?oo
o`0=0?ooo`<0003on03oool001D0oooo00<000000?ooo`3oool0403oool20000ooH0oooo000E0?oo
o`030000003oool0oooo0180oooo0`000?oc0?ooo`005@3oool00`000000oooo0?ooo`0E0?ooo`80
003ol@3oool001D0oooo0P00000H0?ooo`80003ok`3oool001D0oooo00<000000?ooo`3oool06@3o
ool20000ond0oooo000E0?ooo`030000003oool0oooo01/0oooo0P000?o[0?ooo`005@3oool00`00
0000oooo0?ooo`0M0?ooo`<0003oj03oool001D0oooo00<000000?ooo`3oool0803oool20000onH0
oooo000E0?ooo`8000008`3oool30000on<0oooo000E0?ooo`030000003oool0oooo02D0oooo0P00
0?oQ0?ooo`005@3oool00`000000oooo0?ooo`0W0?ooo`<0003ogP3oool001D0oooo00<000000?oo
o`3oool0:P3oool30000om/0oooo00020?ooo`800000103oool2000000<0oooo100000040?ooo`03
0000003oool0oooo02d0oooo0P000?oI0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool0;`3oool30000omH0oooo00001@3oool00000
0?ooo`3oool0000000T0oooo00<000000?ooo`3oool0103oool300000380oooo0`000?oC0?ooo`00
00D0oooo0000003oool0oooo0000000:0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool0=@3oool20000om40oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3o
ool00000103oool00`000000oooo0?ooo`0g0?ooo`80003oc`3oool00080oooo0P00000:0?ooo`80
00001@3oool00`000000oooo0?ooo`0i0?ooo`80003oc@3oool001D0oooo00<000000?ooo`3oool0
>`3oool20000ol/0oooo000E0?ooo`800000?P3oool30000olP0oooo000E0?ooo`030000003oool0
oooo0400oooo0`000?o50?ooo`005@3oool00`000000oooo0?ooo`130?ooo`80003o``3oool001D0
oooo00<000000?ooo`3oool0A@3oool30000ol00oooo000E0?ooo`030000003oool0oooo04P0oooo
0`000?nm0?ooo`005@3oool2000004`0oooo0P000?nk0?ooo`005@3oool00`000000oooo0?ooo`1=
0?ooo`80003o^@3oool001D0oooo00<000000?ooo`3oool0C`3oool20000okL0oooo000E0?ooo`03
0000003oool0oooo0540oooo0P000?ne0?ooo`005@3oool00`000000oooo0?ooo`1C0?ooo`<0003o
/P3oool001D0oooo00<000000?ooo`3oool0EP3oool20000ok00oooo000E0?ooo`800000F@3oool3
0000ojd0oooo000E0?ooo`030000003oool0oooo05/0oooo0P000?n[0?ooo`005@3oool00`000000
oooo0?ooo`1M0?ooo`<0003oZ03oool00080oooo0P0000040?ooo`8000001@3oool3000000<0oooo
00<000000?ooo`3oool0H03oool20000ojH0oooo00001@3oool000000?ooo`3oool0000000/0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`1R0?ooo`80003oY03oool000050?ooo`00
0000oooo0?ooo`000000203oool5000000<0oooo0`00001T0?ooo`80003oXP3oool000050?ooo`00
0000oooo0?ooo`000000203oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo06H0
oooo0P000?nP0?ooo`0000D0oooo0000003oool0oooo000000090?ooo`030000003oool0000000@0
oooo00<000000?ooo`3oool0J03oool30000oid0oooo00020?ooo`8000002`3oool2000000@0oooo
00<000000?ooo`3oool0J`3oool30000oiX0oooo000E0?ooo`030000003oool0oooo06h0oooo0P00
0?nH0?ooo`005@3oool200000740oooo0`000?nE0?ooo`005@3oool00`000000oooo0?ooo`1c0?oo
o`<0003oTP3oool001D0oooo00<000000?ooo`3oool0MP3oool20000oi00oooo000E0?ooo`030000
003oool0oooo07P0oooo0`000?n=0?ooo`005@3oool00`000000oooo0?ooo`1k0?ooo`80003oR`3o
ool001D0oooo00<000000?ooo`3oool0O@3oool30000ohP0oooo000E0?ooo`800000P@3oool20000
ohH0oooo000E0?ooo`030000003oool0oooo0880oooo0P000?n40?ooo`005@3oool00`000000oooo
0?ooo`240?ooo`80003oPP3oool001D0oooo00<000000?ooo`3oool0QP3oool20000oh00oooo000E
0?ooo`030000003oool0oooo08P0oooo0`000?mm0?ooo`005@3oool2000008`0oooo0`000?mj0?oo
o`005@3oool00`000000oooo0?ooo`2>0?ooo`80003oN03oool001D0oooo00<000000?ooo`3oool0
T03oool30000ogD0oooo00020?ooo`800000103oool2000000<0oooo0`0000050?ooo`030000003o
ool0oooo09<0oooo0`000?mb0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003o
ool0oooo000000@0oooo00<000000?ooo`3oool0UP3oool20000og00oooo00001@3oool000000?oo
o`3oool0000000P0oooo00@000000?ooo`3oool00000103oool3000009P0oooo0P000?m^0?ooo`00
00D0oooo0000003oool0oooo000000080?ooo`<000001@3oool00`000000oooo0?ooo`2J0?ooo`80
003oK03oool000050?ooo`000000oooo0?ooo`0000002@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo09`0oooo0P000?mZ0?ooo`000P3oool2000000X0oooo0`0000040?ooo`030000
003oool0oooo09h0oooo0`000?mW0?ooo`005@3oool00`000000oooo0?ooo`2Q0?ooo`<0003oI03o
ool001D0oooo00<000000?ooo`3oool0Y03oool20000of80oooo000E0?ooo`800000Y`3oool30000
oel0oooo000E0?ooo`030000003oool0oooo0:T0oooo0P000?mM0?ooo`005@3oool00`000000oooo
0?ooo`2[0?ooo`80003oF`3oool001D0oooo00<000000?ooo`3oool0[@3oool20000oeT0oooo000E
0?ooo`030000003oool0oooo0:l0oooo0P000?mG0?ooo`005@3oool200000;80oooo0P000?mE0?oo
o`005@3oool00`000000oooo0?ooo`2c0?ooo`<0003oDP3oool001D0oooo00<000000?ooo`3oool0
]P3oool30000odl0oooo000E0?ooo`030000003oool0oooo0;T0oooo0P000?m=0?ooo`005@3oool0
0`000000oooo0?ooo`2k0?ooo`<0003oBP3oool001D0oooo0P00002o0?ooo`<0003oA`3oool001D0
oooo00<000000?ooo`3oool0`@3oool20000odD0oooo000E0?ooo`030000003oool0oooo0<<0oooo
0P000?m30?ooo`005@3oool00`000000oooo0?ooo`350?ooo`80003o@@3oool00080oooo0P000004
0?ooo`800000103oool2000000D0oooo00<000000?ooo`3oool0a`3oool20000ocl0oooo00001@3o
ool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?oo
o`390?ooo`<0003o?03oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?oo
o`0000040?ooo`<00000c03oool30000ocT0oooo00001@3oool000000?ooo`3oool0000000T0oooo
0P0000050?ooo`030000003oool0oooo0<l0oooo0P000?lg0?ooo`0000D0oooo0000003oool0oooo
000000080?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool0d@3oool30000oc@0
oooo00020?ooo`8000002P3oool2000000D0oooo00<000000?ooo`3oool0e03oool30000oc40oooo
000E0?ooo`030000003oool0oooo0=L0oooo0P000?l_0?ooo`005@3oool200000=X0oooo0P000?l]
0?ooo`005@3oool00`000000oooo0?ooo`3K0?ooo`80003o:`3oool001D0oooo00<000000?ooo`3o
ool0g@3oool20000obT0oooo000E0?ooo`030000003oool0oooo0=l0oooo0`000?lV0?ooo`005@3o
ool00`000000oooo0?ooo`3R0?ooo`<0003o8`3oool001D0oooo0P00003V0?ooo`80003o8@3oool0
01D0oooo00<000000?ooo`3oool0i`3oool30000oah0oooo000E0?ooo`030000003oool0oooo0>X0
oooo0`000?lK0?ooo`005@3oool00`000000oooo0?ooo`3]0?ooo`80003o6@3oool001D0oooo00<0
00000?ooo`3oool0k`3oool20000oaL0oooo000E0?ooo`030000003oool0oooo0?40oooo0P000?lE
0?ooo`005@3oool200000?@0oooo0P000?lC0?ooo`005@3oool00`000000oooo0?ooo`3e0?ooo`<0
003o403oool001D0oooo00<000000?ooo`3oool0n03oool20000o`h0oooo000=0?ooo`D000000`3o
ool00`000000oooo0?ooo`3j0?ooo`<0003o2`3oool000l0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`3m0?ooo`80003o2@3oool000l0oooo00<000000?ooo`3oool00`3ooolI0?l0
0>T0oooo0P000?l70?ooo`003`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01H0
oooo603o003J0?ooo`003`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo02h0oooo
2`3o003?0?ooo`003P3oool2000000D0oooo00<000000?ooo`3oool0>@3oool90?l00<H0oooo000E
0?ooo`030000003oool0oooo0480oooo1P3o00300?ooo`005@3oool2000004T0oooo1`3o002i0?oo
o`005@3oool00`000000oooo0?ooo`1?0?ooo`P0o`00/@3oool001D0oooo00<000000?ooo`3oool0
E`3oool60?l00:/0oooo000E0?ooo`030000003oool0oooo05d0oooo203o002S0?ooo`005@3oool0
0`000000oooo0?ooo`1U0?ooo`P0o`00V`3oool001D0oooo00<000000?ooo`3oool0K@3oool50?l0
09H0oooo000E0?ooo`800000L`3oool50?l00940oooo000E0?ooo`030000003oool0oooo07L0oooo
1P3o002;0?ooo`005@3oool00`000000oooo0?ooo`1m0?ooo`@0o`00Q`3oool001D0oooo00<00000
0?ooo`3oool0P@3oool40?l008<0oooo000E0?ooo`030000003oool0oooo08D0oooo103o001o0?oo
o`005@3oool2000008X0oooo103o001k0?ooo`005@3oool00`000000oooo0?ooo`2=0?ooo`H0o`00
M@3oool001D0oooo00<000000?ooo`3oool0T`3oool40?l00740oooo00000`3oool0000000000003
00000080oooo0P0000030?ooo`@00000103oool00`000000oooo0?ooo`2G0?ooo`@0o`00K@3oool0
00<0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
09/0oooo103o001Y0?ooo`000`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00@0
oooo0`00002O0?ooo`<0o`00IP3oool000<0oooo00<000000?ooo`3oool02@3oool00`000000oooo
0?ooo`030?ooo`030000003oool0oooo0:80oooo0`3o001S0?ooo`000`3oool00`000000oooo0?oo
o`070?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool0Y@3oool40?l005l0oooo
00020?ooo`8000002P3oool2000000D0oooo00<000000?ooo`3oool0Z@3oool30?l005`0oooo000E
0?ooo`030000003oool0oooo0:`0oooo103o001H0?ooo`005@3oool00`000000oooo0?ooo`2`0?oo
o`@0o`00E03oool001D0oooo0P00002e0?ooo`@0o`00D03oool001D0oooo00<000000?ooo`3oool0
^03oool30?l004d0oooo000E0?ooo`030000003oool0oooo0;/0oooo103o00190?ooo`005@3oool0
0`000000oooo0?ooo`2o0?ooo`<0o`00AP3oool001D0oooo00<000000?ooo`3oool0`P3oool20?l0
04@0oooo000E0?ooo`800000a@3oool30?l00440oooo000E0?ooo`030000003oool0oooo0<L0oooo
0P3o000o0?ooo`005@3oool00`000000oooo0?ooo`390?ooo`<0o`00?03oool001D0oooo00<00000
0?ooo`3oool0c03oool40?l003P0oooo000E0?ooo`030000003oool0oooo0=00oooo103o000d0?oo
o`005@3oool200000=D0oooo0`3o000a0?ooo`005@3oool00`000000oooo0?ooo`3G0?ooo`<0o`00
;P3oool001D0oooo00<000000?ooo`3oool0fP3oool20?l002`0oooo000E0?ooo`030000003oool0
oooo0=`0oooo0`3o000Y0?ooo`0000<0oooo0000000000000`0000020?ooo`8000001@3oool30000
00<0oooo00<000000?ooo`3oool0g`3oool30?l002H0oooo00030?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3R0?ooo`<0o`008`3oool000<0oooo
00<000000?ooo`3oool01`3oool5000000<0oooo0`00003U0?ooo`80o`008@3oool000<0oooo00<0
00000?ooo`3oool01`3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0>L0oooo
0`3o000N0?ooo`000`3oool00`000000oooo0?ooo`080?ooo`030000003oool0000000@0oooo00<0
00000?ooo`3oool0jP3oool30?l001/0oooo00020?ooo`8000002`3oool2000000@0oooo00<00000
0?ooo`3oool0k@3oool30?l001P0oooo000E0?ooo`030000003oool0oooo0?00oooo0P3o000F0?oo
o`005@3oool200000?<0oooo0`3o000C0?ooo`005@3oool00`000000oooo0?ooo`3e0?ooo`<0o`00
403oool001D0oooo00<000000?ooo`3oool0n03oool20?l000h0oooo000E0?ooo`030000003oool0
oooo0?X0oooo0`3o000;0?ooo`005@3oool00`000000oooo0?ooo`3m0?ooo`80o`002@3oool001D0
oooo0P00003o0?ooo`40oooo0P3o00070?ooo`005@3oool00`000000oooo0?ooo`3o0?ooo`T0oooo
000E0?ooo`030000003oool0oooo0?l0oooo2@3oool001D0oooo00<000000?ooo`3oool0o`3oool9
0?ooo`005@3oool00`000000oooo0?ooo`3o0?ooo`T0oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0825959, -0.117974, \
0.00385924, 0.00936656}}]
}, Open  ]],

Cell[TextData[{
  "In general, you must specify initial conditions for the differential \
variables because typically there is a parametrized general solution. For \
this problem with ",
  Cell[BoxData[
      \(x\_2[t] = \ 0\)]],
  ", the general solution is {",
  Cell[BoxData[
      \(x\_1[t] = \ x\_1[0]\  + \ t\^2/2\)]],
  ",",
  Cell[BoxData[
      \(x\_2[t] = \ 0\)]],
  ",",
  Cell[BoxData[
      \(x\_3[t] == \ t\)]],
  "}, so it is necessary to give ",
  Cell[BoxData[
      \(\(\(\ \)\(x\_1[0]\)\)\)]],
  "to determine the solution."
}], "Text"],

Cell[TextData[{
  StyleBox["NDSolve", "MR"],
  " cannot always find initial conditions consistent with the equations \
because sometimes this is a difficult problem. [",
  ButtonBox["BCP89",
    ButtonData:>{"References.nb", "BCP89"},
    ButtonStyle->"Hyperlink"],
  "] \[OpenCurlyDoubleQuote]Often the most difficult part of solving a DAE \
system in applications is to determine a consistent set of initial conditions \
with which to start the computation.\[CloseCurlyDoubleQuote] "
}], "Text"],

Cell[TextData[{
  StyleBox["NDSolve", "MR"],
  " fails to find a consistent initial condition."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[{DAE, \ x\_1[0]\  \[Equal] \ 1}, \ {x\_1, x\_2, x\_3}, \ {t, 0, 
        1}]\)], "Input",
  CellLabel->"In[12]:="],

Cell[BoxData[
    \(NDSolve::"icfail" \(\(:\)\(\ \)\) 
      "Unable to find initial conditions which satisfy the residual function \
within specified tolerances.  Try giving initial conditions for both values \
and derivatives of the functions."\)], "Message",
  CellLabel->"From In[12]:="],

Cell[BoxData[
    \({}\)], "Output",
  CellLabel->"Out[12]="]
}, Open  ]],

Cell[TextData[{
  "If ",
  StyleBox["NDSolve", "MR"],
  " fails to find consistent initial conditions, you can use ",
  StyleBox["FindRoot", "MR"],
  " with a good starting value or some other procedure to obtain consistent \
initial conditions and supply them. If you know values close to a good \
starting guess, ",
  StyleBox["NDSolve", "MR"],
  " uses these values to start its search, which may help. You may specify \
values of the dependent variables and their derivatives."
}], "Text"],

Cell["\<\
With index-1 systems of DAEs, it is often possible to differentiate and use \
an ODE solver to get the solution. \
\>", "Text"],

Cell["\<\
Here is the Robertson chemical kinetics problem. Because of the large and \
small rate constants, the problem is quite stiff.\
\>", "MathCaption"],

Cell[BoxData[{
    \(kinetics\  = \ {\(y\_1'\)[
            t]\  \[Equal] \(-\(1\/25\)\)\ y\_1[t]\  + \ \(10\^4\) 
              y\_2[t]\ y\_3[t], \ \(y\_2'\)[t]\  \[Equal] \ 
          1\/25\ y\_1[t]\  - \ \ 3\ \(10\^7\) 
              y\_2[t]\^2\ }; \), "\[IndentingNewLine]", 
    \(balance\  = \ 
      y\_1[t]\  + \ y\_2[t]\  + \ y\_3[t]\  \[Equal] \ 
        1; \), "\[IndentingNewLine]", 
    \(start\  = \ {y\_1[0]\ \  \[Equal] \ 1, \ y\_2[0]\  \[Equal] \ 0, \ 
        y\_3[0]\  \[Equal] \ 0}; \)}], "Input",
  CellLabel->"In[13]:="],

Cell["\<\
This solves the Robertson kinetics problem as an ODE by differentiating the \
balance equation.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(odesol\  = \ 
      First[NDSolve[{kinetics, \ D[balance, t], \ start}, \ {y\_1, \ y\_2, \ 
            y\_3}, \ {t, \ 0, \ 40000}]]\)], "Input",
  CellLabel->"In[16]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{\(y\_1\), "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 40000.`}}, "<>"]\),
            False,
            Editable->False]}], ",", 
        RowBox[{\(y\_2\), "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 40000.`}}, "<>"]\),
            False,
            Editable->False]}], ",", 
        RowBox[{\(y\_3\), "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 40000.`}}, "<>"]\),
            False,
            Editable->False]}]}], "}"}]], "Output",
  CellLabel->"Out[16]="]
}, Open  ]],

Cell[TextData[{
  "The stiffness of the problem is supported by ",
  Cell[BoxData[
      \(TraditionalForm\`y\_1\)]],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`y\_2\)]],
  " having their main variation on two completely different time scales."
}], "Text"],

Cell[TextData[{
  "This shows the solutions ",
  Cell[BoxData[
      \(y\_1\)]],
  " and ",
  Cell[BoxData[
      \(y\_2\)]],
  "."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Show[
      Block[{$DisplayFunction\  = \ 
            Identity}, \
\[IndentingNewLine]GraphicsArray[{\[IndentingNewLine]Plot[
              y\_1[t]\  /. \ odesol, {t, 0, 25}, \ PlotRange \[Rule] All], 
            Plot[y\_2[t]\  /. \ odesol, {t, 0, 0.01}, \ 
              PlotRange \[Rule] All]\[IndentingNewLine]}]]]; \)], "Input",
  CellLabel->"In[17]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .47619 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.47619 0.0113379 0.47619 [
[ 0 0 0 0 ]
[ 1 .47619 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .47619 L
0 .47619 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.0113379 0.477324 0.464853 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0380952 -5.07102 5.67434 [
[.21429 .59082 -3 -9 ]
[.21429 .59082 3 0 ]
[.40476 .59082 -6 -9 ]
[.40476 .59082 6 0 ]
[.59524 .59082 -6 -9 ]
[.59524 .59082 6 0 ]
[.78571 .59082 -6 -9 ]
[.78571 .59082 6 0 ]
[.97619 .59082 -6 -9 ]
[.97619 .59082 6 0 ]
[.01131 .03589 -18 -4.5 ]
[.01131 .03589 0 4.5 ]
[.01131 .14937 -24 -4.5 ]
[.01131 .14937 0 4.5 ]
[.01131 .26286 -24 -4.5 ]
[.01131 .26286 0 4.5 ]
[.01131 .37635 -24 -4.5 ]
[.01131 .37635 0 4.5 ]
[.01131 .48983 -24 -4.5 ]
[.01131 .48983 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .60332 m
.21429 .60957 L
s
[(5)] .21429 .59082 0 1 Mshowa
.40476 .60332 m
.40476 .60957 L
s
[(10)] .40476 .59082 0 1 Mshowa
.59524 .60332 m
.59524 .60957 L
s
[(15)] .59524 .59082 0 1 Mshowa
.78571 .60332 m
.78571 .60957 L
s
[(20)] .78571 .59082 0 1 Mshowa
.97619 .60332 m
.97619 .60957 L
s
[(25)] .97619 .59082 0 1 Mshowa
.125 Mabswid
.0619 .60332 m
.0619 .60707 L
s
.1 .60332 m
.1 .60707 L
s
.1381 .60332 m
.1381 .60707 L
s
.17619 .60332 m
.17619 .60707 L
s
.25238 .60332 m
.25238 .60707 L
s
.29048 .60332 m
.29048 .60707 L
s
.32857 .60332 m
.32857 .60707 L
s
.36667 .60332 m
.36667 .60707 L
s
.44286 .60332 m
.44286 .60707 L
s
.48095 .60332 m
.48095 .60707 L
s
.51905 .60332 m
.51905 .60707 L
s
.55714 .60332 m
.55714 .60707 L
s
.63333 .60332 m
.63333 .60707 L
s
.67143 .60332 m
.67143 .60707 L
s
.70952 .60332 m
.70952 .60707 L
s
.74762 .60332 m
.74762 .60707 L
s
.82381 .60332 m
.82381 .60707 L
s
.8619 .60332 m
.8619 .60707 L
s
.9 .60332 m
.9 .60707 L
s
.9381 .60332 m
.9381 .60707 L
s
.25 Mabswid
0 .60332 m
1 .60332 L
s
.02381 .03589 m
.03006 .03589 L
s
[(0.9)] .01131 .03589 1 0 Mshowa
.02381 .14937 m
.03006 .14937 L
s
[(0.92)] .01131 .14937 1 0 Mshowa
.02381 .26286 m
.03006 .26286 L
s
[(0.94)] .01131 .26286 1 0 Mshowa
.02381 .37635 m
.03006 .37635 L
s
[(0.96)] .01131 .37635 1 0 Mshowa
.02381 .48983 m
.03006 .48983 L
s
[(0.98)] .01131 .48983 1 0 Mshowa
.125 Mabswid
.02381 .06426 m
.02756 .06426 L
s
.02381 .09263 m
.02756 .09263 L
s
.02381 .121 m
.02756 .121 L
s
.02381 .17774 m
.02756 .17774 L
s
.02381 .20612 m
.02756 .20612 L
s
.02381 .23449 m
.02756 .23449 L
s
.02381 .29123 m
.02756 .29123 L
s
.02381 .3196 m
.02756 .3196 L
s
.02381 .34797 m
.02756 .34797 L
s
.02381 .40472 m
.02756 .40472 L
s
.02381 .43309 m
.02756 .43309 L
s
.02381 .46146 m
.02756 .46146 L
s
.02381 .5182 m
.02756 .5182 L
s
.02381 .54658 m
.02756 .54658 L
s
.02381 .57495 m
.02756 .57495 L
s
.02381 .00751 m
.02756 .00751 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.5 Mabswid
.02381 .60332 m
.04262 .50167 L
.06244 .41411 L
.08426 .33636 L
.10458 .27793 L
.14264 .19598 L
.16305 .16325 L
.18163 .1387 L
.20239 .11607 L
.22156 .09889 L
.25997 .07277 L
.2787 .06316 L
.29932 .05441 L
.32036 .04712 L
.33961 .04163 L
.37838 .03323 L
.39737 .03013 L
.41809 .02734 L
.43941 .02499 L
.45874 .02324 L
.49787 .02055 L
.51711 .01955 L
.53793 .01867 L
.55748 .01798 L
.57894 .01736 L
.61842 .01651 L
.63793 .01619 L
.65885 .01591 L
.67866 .01569 L
.70021 .0155 L
.74005 .01523 L
.75983 .01513 L
.78084 .01504 L
.81859 .01493 L
.83793 .01488 L
.85879 .01484 L
.87844 .01481 L
.89993 .01478 L
.91909 .01476 L
.93955 .01474 L
.97619 .01472 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.522676 0.0113379 0.97619 0.464853 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 95.2381 0.0147148 16121.4 [
[.21429 .00221 -15 -9 ]
[.21429 .00221 15 0 ]
[.40476 .00221 -15 -9 ]
[.40476 .00221 15 0 ]
[.59524 .00221 -15 -9 ]
[.59524 .00221 15 0 ]
[.78571 .00221 -15 -9 ]
[.78571 .00221 15 0 ]
[.97619 .00221 -12 -9 ]
[.97619 .00221 12 0 ]
[.01131 .09532 -33 -6.25 ]
[.01131 .09532 0 6.25 ]
[.01131 .17593 -42 -4.5 ]
[.01131 .17593 0 4.5 ]
[.01131 .25654 -48 -4.5 ]
[.01131 .25654 0 4.5 ]
[.01131 .33714 -42 -4.5 ]
[.01131 .33714 0 4.5 ]
[.01131 .41775 -48 -4.5 ]
[.01131 .41775 0 4.5 ]
[.01131 .49836 -42 -4.5 ]
[.01131 .49836 0 4.5 ]
[.01131 .57896 -48 -4.5 ]
[.01131 .57896 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .01471 m
.21429 .02096 L
s
[(0.002)] .21429 .00221 0 1 Mshowa
.40476 .01471 m
.40476 .02096 L
s
[(0.004)] .40476 .00221 0 1 Mshowa
.59524 .01471 m
.59524 .02096 L
s
[(0.006)] .59524 .00221 0 1 Mshowa
.78571 .01471 m
.78571 .02096 L
s
[(0.008)] .78571 .00221 0 1 Mshowa
.97619 .01471 m
.97619 .02096 L
s
[(0.01)] .97619 .00221 0 1 Mshowa
.125 Mabswid
.07143 .01471 m
.07143 .01846 L
s
.11905 .01471 m
.11905 .01846 L
s
.16667 .01471 m
.16667 .01846 L
s
.2619 .01471 m
.2619 .01846 L
s
.30952 .01471 m
.30952 .01846 L
s
.35714 .01471 m
.35714 .01846 L
s
.45238 .01471 m
.45238 .01846 L
s
.5 .01471 m
.5 .01846 L
s
.54762 .01471 m
.54762 .01846 L
s
.64286 .01471 m
.64286 .01846 L
s
.69048 .01471 m
.69048 .01846 L
s
.7381 .01471 m
.7381 .01846 L
s
.83333 .01471 m
.83333 .01846 L
s
.88095 .01471 m
.88095 .01846 L
s
.92857 .01471 m
.92857 .01846 L
s
.25 Mabswid
0 .01471 m
1 .01471 L
s
.02381 .09532 m
.03006 .09532 L
s
gsave
.01131 .09532 -33 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-6) show
grestore
.02381 .17593 m
.03006 .17593 L
s
[(0.00001)] .01131 .17593 1 0 Mshowa
.02381 .25654 m
.03006 .25654 L
s
[(0.000015)] .01131 .25654 1 0 Mshowa
.02381 .33714 m
.03006 .33714 L
s
[(0.00002)] .01131 .33714 1 0 Mshowa
.02381 .41775 m
.03006 .41775 L
s
[(0.000025)] .01131 .41775 1 0 Mshowa
.02381 .49836 m
.03006 .49836 L
s
[(0.00003)] .01131 .49836 1 0 Mshowa
.02381 .57896 m
.03006 .57896 L
s
[(0.000035)] .01131 .57896 1 0 Mshowa
.125 Mabswid
.02381 .03084 m
.02756 .03084 L
s
.02381 .04696 m
.02756 .04696 L
s
.02381 .06308 m
.02756 .06308 L
s
.02381 .0792 m
.02756 .0792 L
s
.02381 .11144 m
.02756 .11144 L
s
.02381 .12756 m
.02756 .12756 L
s
.02381 .14369 m
.02756 .14369 L
s
.02381 .15981 m
.02756 .15981 L
s
.02381 .19205 m
.02756 .19205 L
s
.02381 .20817 m
.02756 .20817 L
s
.02381 .22429 m
.02756 .22429 L
s
.02381 .24041 m
.02756 .24041 L
s
.02381 .27266 m
.02756 .27266 L
s
.02381 .28878 m
.02756 .28878 L
s
.02381 .3049 m
.02756 .3049 L
s
.02381 .32102 m
.02756 .32102 L
s
.02381 .35326 m
.02756 .35326 L
s
.02381 .36939 m
.02756 .36939 L
s
.02381 .38551 m
.02756 .38551 L
s
.02381 .40163 m
.02756 .40163 L
s
.02381 .43387 m
.02756 .43387 L
s
.02381 .44999 m
.02756 .44999 L
s
.02381 .46611 m
.02756 .46611 L
s
.02381 .48224 m
.02756 .48224 L
s
.02381 .51448 m
.02756 .51448 L
s
.02381 .5306 m
.02756 .5306 L
s
.02381 .54672 m
.02756 .54672 L
s
.02381 .56284 m
.02756 .56284 L
s
.02381 .59508 m
.02756 .59508 L
s
.02381 .61121 m
.02756 .61121 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.5 Mabswid
.02381 .01472 m
.04262 .13999 L
.06244 .26036 L
.08426 .3688 L
.10458 .44462 L
.12297 .49423 L
.13327 .51552 L
.14264 .53151 L
.16178 .55611 L
.17259 .56619 L
.18257 .57362 L
.20129 .58385 L
.21194 .58802 L
.22192 .59113 L
.2323 .5937 L
.24216 .59566 L
.26069 .59833 L
.27027 .59933 L
.28049 .60018 L
.29007 .60082 L
.29888 .6013 L
.30917 .60174 L
.32029 .6021 L
.3301 .60235 L
.34046 .60255 L
.34994 .6027 L
.3601 .60283 L
.37844 .603 L
.38861 .60307 L
.39968 .60313 L
.41013 .60317 L
.4198 .6032 L
.43846 .60325 L
.44909 .60327 L
.45906 .60328 L
.46889 .60329 L
.47925 .6033 L
.48808 .6033 L
.49775 .60331 L
.50802 .60331 L
.51736 .60331 L
.52253 .60331 L
.52814 .60332 L
.53339 .60332 L
.53831 .60332 L
.54326 .60332 L
.54778 .60332 L
.55035 .60332 L
.55279 .60332 L
.55537 .60332 L
Mistroke
.55819 .60332 L
.57923 .60332 L
.58939 .60332 L
.59894 .60332 L
.60424 .60332 L
.60569 .60332 L
.60722 .60332 L
.60854 .60332 L
.60995 .60332 L
.61124 .60332 L
.61242 .60332 L
.6136 .60332 L
.61471 .60332 L
.61601 .60332 L
.61721 .60332 L
.61853 .60332 L
.61993 .60332 L
.62232 .60332 L
.62483 .60332 L
.62698 .60332 L
.62932 .60332 L
.63457 .60332 L
.63944 .60332 L
.64829 .60332 L
.65773 .60332 L
.668 .60332 L
.67912 .60331 L
.69928 .60331 L
.71919 .60331 L
.74025 .6033 L
.77819 .6033 L
.81858 .60329 L
.85746 .60329 L
.89879 .60328 L
.9386 .60328 L
.97619 .60327 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[17]:=",
  ImageSize->{495.813, 236.125},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40007_0000k2000`400?l00000o`00003oo`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`
0?ooo`009P3oool00`000000oooo0?ooo`3o0?ooolL0oooo000V0?ooo`800000o`3oooo80?ooo`00
9P3oool00`000000oooo0?ooo`3o0?ooolL0oooo000V0?ooo`030000003oool0oooo06P0ooooEP00
003o0?ooo`T0oooo000V0?ooo`030000003oool0oooo05X0oooo3P00003o0?oooel0oooo000D0?oo
o`800000103oool2000000<0oooo0`0000040?ooo`030000003oool0oooo04/0oooo3`00003o0?oo
ofd0oooo000C0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`170?ooo`@00000e03oool2000000@0oooo0P0000040?ooo`800000103oool20000
00<0oooo100000050?ooo`800000103oool2000000@0oooo0P0000040?ooo`8000001@3oool30000
00@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000030?ooo`<000001`3oool20000
00@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000090?ooo`800000103oool20000
00@0oooo0P0000030?ooo`D000003@3oool001<0oooo00@000000?ooo`3oool000002@3oool30000
00<0oooo0`0000120?ooo`D00000e`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0
oooo00000080oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`050?ooo`040000
003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`000005
0?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool00000203oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000001@3oool01000
0000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000
0P3oool010000000oooo0?ooo`0000070?ooo`040000003oool0oooo000000P0oooo00@000000?oo
o`3oool00000103oool00`000000oooo0?ooo`0=0?ooo`004`3oool010000000oooo0?ooo`000008
0?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool0?03oool600000=`0oooo00@0
00000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
00<0oooo00<000000?ooo`3oool0103oool010000000oooo0?ooo`0000080?ooo`040000003oool0
oooo00000080oooo00@000000?ooo`3oool000000P3oool5000000<0oooo00@000000?ooo`3oool0
0000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@00000
0?ooo`3oool000001@3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000070?ooo`040000003oool0
oooo000000P0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`0=0?ooo`004`3o
ool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3o
ool0>@3oool300000>80oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool00`3oool010000000oooo0?oo
o`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool01000
0000oooo0?ooo`0000040?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool00000
0P3oool010000000oooo0?ooo`0000020?ooo`<000001P3oool010000000oooo0?ooo`0000080?oo
o`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000`3oool2000000P0oooo00@0
00000?ooo`3oool00000203oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo00d0
oooo000D0?ooo`8000002P3oool3000000<0oooo00<000000?ooo`3oool0=`3oool200000>D0oooo
00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
00000080oooo00@000000?ooo`3oool00000103oool010000000oooo0?ooo`0000080?ooo`040000
003oool0oooo00000080oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000040?oo
o`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?oo
o`0000030?ooo`030000003oool0oooo00D0oooo00@000000?ooo`3oool00000203oool010000000
oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000001`3o
ool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3o
ool03@3oool002H0oooo00<000000?ooo`3oool0=03oool300000>P0oooo0P00000:0?ooo`800000
103oool2000000@0oooo0P0000060?ooo`8000002P3oool2000000@0oooo0P0000050?ooo`800000
1@3oool2000000X0oooo0P0000040?ooo`800000103oool3000000H0oooo0P00000:0?ooo`800000
103oool2000000@0oooo0P0000090?ooo`8000002P3oool2000000@0oooo0P00000?0?ooo`009P3o
ool00`000000oooo0?ooo`0b0?ooo`800000o`3ooonC0?ooo`009P3oool200000300oooo0`00003o
0?oooiD0oooo000V0?ooo`030000003oool0oooo02d0oooo0P00003L0?ooo`030000003oool0oooo
0;P0oooo000V0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool0g03oool00`000000
oooo0?ooo`2h0?ooo`009P3oool00`000000oooo0?ooo`0Z0?ooo`800000f`3ooon[000001@0oooo
000V0?ooo`030000003oool0oooo02T0oooo00<000000?ooo`3oool0g`3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?oo
o`060?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05P3oool002H0oooo0P00000Y
0?ooo`030000003oool0oooo0>00oooo00<000000?ooo`3oool0^03oool002H0oooo00<000000?oo
o`3oool09P3oool200000><0oooo0P00002i0?ooo`009P3oool00`000000oooo0?ooo`0U0?ooo`03
0000003oool0oooo0><0oooo0P00002i0?ooo`009P3oool00`000000oooo0?ooo`0T0?ooo`030000
003oool0oooo0>@0oooo0P00002i0?ooo`009P3oool00`000000oooo0?ooo`0R0?ooo`800000i`3o
ool200000;T0oooo000V0?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool0i`3oool2
00000;T0oooo000V0?ooo`8000008@3oool00`000000oooo0?ooo`3X0?ooo`800000^@3oool002H0
oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`3Y0?ooo`800000^@3oool002H0oooo
00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`3Y0?ooo`800000^@3oool002H0oooo00<0
00000?ooo`3oool07P3oool00`000000oooo0?ooo`390?ooo`8000000`3oool01@000000oooo0?oo
o`3oool0000000@0oooo1@0000020?ooo`8000002P3oool200000;T0oooo000>0?ooo`800000103o
ool2000000<0oooo0`0000030?ooo`@000000`3oool00`000000oooo0?ooo`0M0?ooo`030000003o
ool0oooo0<T0oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000070?ooo`050000
003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`070?ooo`030000003oool000000;P0
oooo000=0?ooo`040000003oool0oooo000000X0oooo00D000000?ooo`3oool0oooo000000060?oo
o`030000003oool0oooo01`0oooo00<000000?ooo`3oool0c@3oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0
oooo00L0oooo00<000000?ooo`000000^03oool000d0oooo00@000000?ooo`3oool000002@3oool3
000000<0oooo00<000000?ooo`3oool00`3oool3000001/0oooo00<000000?ooo`3oool0c03oool2
000000@0oooo00<000000?ooo`0000001`3oool01@000000oooo0?ooo`3oool000000080oooo00<0
00000?ooo`3oool01`3oool300000;P0oooo000=0?ooo`040000003oool0oooo000000P0oooo00@0
00000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01/0oooo
00<000000?ooo`3oool0c03oool00`000000oooo0?ooo`020?ooo`050000003oool0oooo0?ooo`00
00001P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0000002h0?ooo`003@3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo
00000080oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`0J0?ooo`030000003o
ool0oooo0<d0oooo0`00000<0?ooo`800000103oool2000000P0oooo0`0000000`3oool000000?oo
o`2g0?ooo`003P3oool2000000X0oooo0`0000030?ooo`800000103oool00`000000oooo0?ooo`0I
0?ooo`030000003oool0oooo0>d0oooo00<000000?ooo`0000000P00002h0?ooo`009P3oool00`00
0000oooo0?ooo`0H0?ooo`030000003oool0oooo0>T0oooo0`0000020?ooo`<0000000<0oooo0000
003oool0]`3oool002H0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`3^0?ooo`06
0000003oool000000000003oool00000]`3oool002H0oooo0P00000H0?ooo`030000003oool0oooo
0?00oooo0P0000020?ooo`030000003oool0oooo0;D0oooo000V0?ooo`030000003oool0oooo01H0
oooo00<000000?ooo`3oool0lP3oool010000000oooo0?ooo`00002g0?ooo`009P3oool00`000000
oooo0?ooo`0F0?ooo`030000003oool0oooo0?80oooo0P0000000`3oool000000?ooo`2f0?ooo`00
9P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0?<0oooo00@000000?ooo`3oool0
0000]`3oool002H0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`3c0?ooo`040000
003oool0oooo00000;L0oooo000V0?ooo`8000005@3oool00`000000oooo0?ooo`3d0?ooo`800000
0P3oool00`000000oooo0?ooo`2d0?ooo`009P3oool00`000000oooo0?ooo`0D0?ooo`030000003o
ool0oooo0<X0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`800000103o
ool2000000<0oooo1@0000020?ooo`050000003oool0oooo0?ooo`000000]P3oool002H0oooo00<0
00000?ooo`3oool04`3oool00`000000oooo0?ooo`3:0?ooo`040000003oool0oooo000000P0oooo
00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
00000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`050000003o
ool0oooo0?ooo`000000]P3oool002H0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?oo
o`3:0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000
oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103o
ool00`000000oooo0?ooo`020?ooo`8000000P3oool00`000000oooo0?ooo`2d0?ooo`009P3oool0
0`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0</0oooo00@000000?ooo`3oool00000203o
ool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3o
ool000000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0080oooo00D00000
0?ooo`3oool0oooo0000002f0?ooo`009P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0
oooo0</0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`040000
003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`000004
0?ooo`030000003oool0oooo0080oooo0P0000030?ooo`030000003oool0oooo0;<0oooo000V0?oo
o`8000004P3oool00`000000oooo0?ooo`3=0?ooo`8000002P3oool2000000@0oooo0P0000040?oo
o`800000103oool2000000@0oooo0P0000040?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool0/`3oool002H0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`3g0?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool0/`3oool002H0oooo00<000000?ooo`3oool0
403oool00`000000oooo0?ooo`3h0?ooo`8000000`3oool00`000000oooo0?ooo`2c0?ooo`009P3o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0?P0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`2c0?ooo`003P3oool2000000@0oooo0P0000030?ooo`<000001@3oool3
00000080oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`3i0?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool0/P3oool000d0oooo00@000000?ooo`3oool000002P3oool0
0`000000oooo0?ooo`040?ooo`050000003oool0oooo0?ooo`0000004@3oool00`000000oooo0?oo
o`3i0?ooo`800000103oool00`000000oooo0?ooo`2b0?ooo`003@3oool010000000oooo0?ooo`00
00090?ooo`<000000P3oool500000080oooo0`00000>0?ooo`030000003oool0oooo0?X0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`2b0?ooo`003@3oool010000000oooo0?ooo`00
00080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000`3oool00`000000
oooo0?ooo`0>0?ooo`030000003oool0oooo0?X0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`2b0?ooo`003@3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo0000
00<0oooo00<000000?ooo`0000000`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo
0</0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`800000103oool20000
00<0oooo1@0000020?ooo`800000103oool2000000@0oooo00<000000?ooo`3oool0/P3oool000h0
oooo0P00000:0?ooo`<00000103oool2000000<0oooo00<000000?ooo`3oool03@3oool00`000000
oooo0?ooo`3:0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool0
10000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
0000103oool01@000000oooo0?ooo`3oool000000080oooo00D000000?ooo`3oool0oooo00000006
0?ooo`030000003oool0oooo0;40oooo000V0?ooo`030000003oool0oooo00d0oooo00<000000?oo
o`3oool0bP3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
00@0oooo00<000000?ooo`3oool0103oool010000000oooo0?ooo`3oool2000000D0oooo00<00000
0?ooo`3oool0/@3oool002H0oooo0P00000=0?ooo`030000003oool0oooo0</0oooo00@000000?oo
o`3oool00000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo
00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo
0080oooo0P0000040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0/@3oool002H0
oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`3;0?ooo`040000003oool0oooo0000
00P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003o
ool0oooo00000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`2a0?ooo`00
9P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0<`0oooo0P00000:0?ooo`800000
103oool2000000@0oooo0P0000040?ooo`800000103oool2000000@0oooo0`0000030?ooo`800000
1@3oool00`000000oooo0?ooo`2a0?ooo`009P3oool00`000000oooo0?ooo`0;0?ooo`030000003o
ool0oooo0?d0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`2a0?ooo`009P3oool0
0`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0?d0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`2`0?ooo`009P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo
0?h0oooo0P0000060?ooo`030000003oool0oooo0;00oooo000V0?ooo`8000002`3oool00`000000
oooo0?ooo`3n0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0/03oool002H0oooo
00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`3n0?ooo`030000003oool0oooo00D0oooo
00<000000?ooo`3oool0/03oool002H0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?oo
o`3o0?ooo`8000001P3oool00`000000oooo0?ooo`2`0?ooo`009P3oool00`000000oooo0?ooo`09
0?ooo`030000003oool0oooo0?l0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`2`
0?ooo`009P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0?l0oooo0P0000070?oo
o`030000003oool0oooo0:l0oooo000V0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool0eP3oool2000000@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`80
00000`3oool4000000<0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`2_0?ooo`00
9P3oool2000000T0oooo00<000000?ooo`3oool0e@3oool010000000oooo0?ooo`0000080?ooo`04
0000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo0:l0oooo000V0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0e@3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0
oooo000000<0oooo00<000000?ooo`3oool00`3oool2000000L0oooo00<000000?ooo`3oool0[`3o
ool002H0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`3E0?ooo`040000003oool0
oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`04
0000003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0[`3oool002H0oooo00<000000?oo
o`3oool01`3oool00`000000oooo0?ooo`3F0?ooo`040000003oool0oooo000000P0oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000030?ooo`030000003oool0
oooo00H0oooo00<000000?ooo`3oool0[`3oool000h0oooo0P0000040?ooo`8000000`3oool30000
00<0oooo0`0000040?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0e`3oool20000
00X0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`800000103oool20000
00P0oooo00<000000?ooo`3oool0[P3oool000d0oooo00@000000?ooo`3oool000002P3oool01@00
0000oooo0?ooo`3oool000000080oooo00D000000?ooo`3oool0oooo000000090?ooo`030000003o
ool0oooo0?l0oooo0P3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0:h0oooo000=
0?ooo`040000003oool0oooo000000T0oooo0`0000020?ooo`040000003oool0oooo000000<0oooo
0`0000070?ooo`030000003oool0oooo0?l0oooo0P3oool00`000000oooo0?ooo`070?ooo`030000
003oool0oooo0:h0oooo000=0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
00000P3oool3000000@0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`3o0?ooo`<0
oooo0P0000080?ooo`030000003oool0oooo0:h0oooo000=0?ooo`040000003oool0oooo000000P0
oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
00H0oooo00<000000?ooo`3oool0o`3oool30?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool0[@3oool000h0oooo0P00000:0?ooo`<000000`3oool3000000<0oooo00<000000?ooo`3o
ool01P3oool00`000000oooo0?ooo`3o0?ooo`<0oooo0P0000090?ooo`030000003oool0oooo0:d0
oooo000V0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0o`3oool30?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool0[@3oool002H0oooo0P0000060?ooo`030000003o
ool0oooo0?l0oooo103oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0:`0oooo000V
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0o`3oool40?ooo`8000002P3oool0
0`000000oooo0?ooo`2/0?ooo`009P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
0=<0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`800000103oool20000
00<0oooo100000030?ooo`800000103oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo
0:`0oooo000V0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0dP3oool010000000
oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3o
ool00`3oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool0[03oool002H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`3B0?ooo`04
0000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000`3oool00`000000
oooo0?ooo`050?ooo`040000003oool0oooo0?ooo`8000002`3oool00`000000oooo0?ooo`2[0?oo
o`009P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0=<0oooo00@000000?ooo`3o
ool00000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0080
oooo0P0000040?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0Z`3oool002H0oooo
0P0000050?ooo`030000003oool0oooo0=<0oooo00@000000?ooo`3oool00000203oool010000000
oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3o
ool00`3oool2000000/0oooo00<000000?ooo`3oool0Z`3oool002H0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`3D0?ooo`8000002P3oool2000000@0oooo0P0000040?ooo`800000
103oool2000000@0oooo0P0000040?ooo`<000000`3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo0:X0oooo000V0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0o`3o
ool50?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0ZP3oool002H0oooo00<00000
0?ooo`3oool00`3oool00`000000oooo0?ooo`3o0?ooo`H0oooo0P00000<0?ooo`030000003oool0
oooo0:X0oooo000V0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0o`3oool60?oo
o`030000003oool0oooo00`0oooo00<000000?ooo`3oool0Z@3oool002H0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`3o0?ooo`H0oooo00<000000?ooo`3oool0303oool00`000000
oooo0?ooo`2Y0?ooo`009P3oool2000000@0oooo00<000000?ooo`3oool0o`3oool60?ooo`800000
3@3oool00`000000oooo0?ooo`2Y0?ooo`009P3oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo0?l0oooo1P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo0:P0oooo000V
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0o`3oool70?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool0Z03oool000h0oooo0P0000040?ooo`8000000`3oool30000
00@0oooo0P0000040?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0g03oool20000
00@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`800000103oool20000
00@0oooo0P00000>0?ooo`030000003oool0oooo0:P0oooo000=0?ooo`040000003oool0oooo0000
00X0oooo00D000000?ooo`3oool0oooo000000020?ooo`050000003oool0oooo0?ooo`000000103o
ool00`000000oooo0?ooo`3K0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00h0oooo
00<000000?ooo`3oool0Y`3oool000d0oooo00@000000?ooo`3oool000002@3oool300000080oooo
00@000000?ooo`3oool000000`3oool300000080oooo00<000000?ooo`3oool0f`3oool010000000
oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3o
ool00P3oool2000000l0oooo00<000000?ooo`3oool0Y`3oool000d0oooo00@000000?ooo`3oool0
0000203oool010000000oooo0?ooo`0000030?ooo`800000103oool01@000000oooo0?ooo`3oool0
00000=h0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`040000
003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`000005
0?ooo`050000003oool0oooo0?ooo`0000004@3oool00`000000oooo0?ooo`2V0?ooo`003@3oool0
10000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000`3oool01@000000oooo0?ooo`3oool000000=h0oooo00@000000?ooo`3oool00000203oool0
10000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000<0oooo00<00000
0?ooo`3oool03`3oool00`000000oooo0?ooo`2V0?ooo`003P3oool2000000X0oooo0`0000030?oo
o`800000103oool01@000000oooo0?ooo`3oool000000=l0oooo0P00000:0?ooo`800000103oool2
000000@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P00000A0?ooo`030000003oool0
oooo0:D0oooo000V0?ooo`050000003oool0oooo0?ooo`000000o`3oool:0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool0Y@3oool002H0oooo00D000000?ooo`3oool0oooo0000003o
0?ooo`X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`2T0?ooo`009P3oool20000
00030?ooo`000000oooo0?l0oooo2P3oool200000180oooo00<000000?ooo`3oool0Y03oool002H0
oooo00@000000?ooo`3oool00000o`3oool;0?ooo`030000003oool0oooo0180oooo00<000000?oo
o`3oool0X`3oool002H0oooo00@000000?ooo`3oool00000o`3oool;0?ooo`030000003oool0oooo
01<0oooo00<000000?ooo`3oool0XP3oool002H0oooo00@000000?ooo`3oool00000o`3oool;0?oo
o`8000005@3oool00`000000oooo0?ooo`2Q0?ooo`009P3oool010000000oooo0?ooo`00003o0?oo
o`/0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`2P0?ooo`009P3oool00`000000
oooo0000003o0?ooo``0oooo0P00000F0?ooo`030000003oool0oooo0:00oooo000V0?ooo`<00000
f`3oool2000000@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`800000
103oool2000000@0oooo0P0000040?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0
W`3oool002H0oooo00<000000?ooo`0000008`3oool200000200oooo1@0000020?ooo`8000007@3o
ool500000080oooo0P00000M0?ooo`@000000`3oool2000001h0oooo100000030?ooo`8000006@3o
ool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3o
ool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool000000P3oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo01L0
oooo0P00002O0?ooo`009P3oool00`000000oooo0000000R0?ooo`040000003oool0oooo00000240
oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo01`0oooo00D000000?oo
o`3oool0oooo000000020?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool00`3oool0
10000000oooo0?ooo`00000M0?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool00000
603oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo
00<000000?ooo`3oool0103oool010000000oooo0?ooo`3oool2000001X0oooo0P00002M0?ooo`00
9P3oool00`000000oooo0000000U0?ooo`030000003oool0oooo01l0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`0K0?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool000007P3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00@000000?ooo`3oool00000203oool0
10000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000050?ooo`040000003oool0oooo0?ooo`800000103oool0
0`000000oooo0?ooo`0K0?ooo`030000003oool0oooo09X0oooo000V0?ooo`800000903oool20000
0280oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo01`0oooo00<00000
0?ooo`3oool00P3oool2000001l0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003o
ool0oooo01d0oooo00<000000?ooo`3oool00P3oool2000001T0oooo00@000000?ooo`3oool00000
203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000<0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0L0?ooo`@00000V03oool002H0oooo0P00
000T0?ooo`030000003oool0oooo0240oooo00D000000?ooo`3oool0oooo000000020?ooo`030000
003oool0oooo01`0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0L0?ooo`040000
003oool0oooo00000080oooo00@000000?ooo`3oool000007@3oool010000000oooo0?ooo`000003
0?ooo`030000003oool0oooo01T0oooo0P00000:0?ooo`800000103oool2000000@0oooo0P000004
0?ooo`800000103oool2000000@0oooo0`0000030?ooo`8000008@3oool600000980oooo000V0?oo
o`800000903oool300000200oooo0P0000040?ooo`8000007P3oool2000000@0oooo0`00000M0?oo
o`800000103oool2000001l0oooo0P0000040?ooo`<00000B@3oool00`000000oooo0?ooo`0V0?oo
ogX00000603oool002H0oooo0P00003o0?ooo`d0oooo00<000000?ooo`3oool0^03oool002H0oooo
0P00003o0?ooo`d0oooo0P00002i0?ooo`009P3oool00`000000oooo0?ooo`3o0?ooo``0oooo00<0
00000?ooo`3oool0^03oool002H0oooo00<000000?ooo`3oool0o`3oooo70?ooo`008@3oooo:0000
0?l0oooo1@3oool002H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`3o0?ooo`L0
oooo000V0?ooo`030000003oool0oooo0?l0ooooa`3oool002H0oooo00<000000?ooo`3oool0o`3o
ooo70?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3oooo`0?ooo`00o`3o
ooo`0?ooo`00\
\>"],
  ImageRangeCache->{{{0, 494.813}, {235.125, 0}} -> {-0.0522423, -0.0238146, \
0.0042531, 0.0042531}, {{12.25, 236.188}, {229.5, 5.5625}} -> {-5.06252, \
0.845028, 0.130601, 0.0008768}, {{258.563, 482.5}, {229.5, 5.5625}} -> \
{-0.0190769, -2.62511*^-05, 6.18*^-05, 3.65087*^-07}}]
}, Open  ]],

Cell["This solves the Robertson kinetics problem as a DAE.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(daesol\  = \ 
      First[NDSolve[{kinetics, \ balance, \ start}, \ {y\_1, \ y\_2, \ 
            y\_3}, \ {t, \ 0, \ 40000}]]\)], "Input",
  CellLabel->"In[18]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{\(y\_1\), "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 40000.`}}, "<>"]\),
            False,
            Editable->False]}], ",", 
        RowBox[{\(y\_2\), "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 40000.`}}, "<>"]\),
            False,
            Editable->False]}], ",", 
        RowBox[{\(y\_3\), "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 40000.`}}, "<>"]\),
            False,
            Editable->False]}]}], "}"}]], "Output",
  CellLabel->"Out[18]="]
}, Open  ]],

Cell["\<\
The solutions for a given component will appear quite close, but comparing \
the chemical balance constraint shows a difference between them.\
\>", "Text"],

Cell[TextData[{
  "Here is a graph of the error in the balance equation for the ODE and DAE \
solutions, shown in black and blue respectively. A log-log scale is used \
because of the large variation in ",
  StyleBox["t",
    FontSlant->"Italic"],
  " and the magnitude of the error."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(berr[t_]\  = \ 
      Abs[Apply[Subtract, balance]]; \), "\[IndentingNewLine]", 
    \(gode\  = \ 
      First[InterpolatingFunctionCoordinates[
          y\_1\  /. \ odesol]]; \), "\[IndentingNewLine]", 
    \(gdae\  = \ 
      First[InterpolatingFunctionCoordinates[
          y\_1\  /. \ daesol]]; \), "\[IndentingNewLine]", 
    \(Show[
      Block[{$DisplayFunction\  = \ 
            Identity}, {\[IndentingNewLine]LogLogListPlot[
            Transpose[{gode, \ 
                berr[gode]\  /. \ 
                  odesol}]], \[IndentingNewLine]LogLogListPlot[
            Transpose[{gdae, \ berr[gdae]\  /. \ daesol}], \ 
            PlotStyle \[Rule] RGBColor[0, 0, 1]]\[IndentingNewLine]}], \ 
      PlotRange \[Rule] All]; \)}], "Input",
  CellLabel->"In[19]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.63169 0.074858 0.966177 0.0364029 [
[.03283 .31203 -23.25 -13.375 ]
[.03283 .31203 23.25 0 ]
[.2574 .31203 -23 -12.5625 ]
[.2574 .31203 23 0 ]
[.48197 .31203 -14 -12.5625 ]
[.48197 .31203 14 0 ]
[.70655 .31203 -8 -12.5625 ]
[.70655 .31203 8 0 ]
[.93112 .31203 -17 -12.5625 ]
[.93112 .31203 17 0 ]
[-0.0125 .09251 -50.6875 -6.6875 ]
[-0.0125 .09251 0 6.6875 ]
[-0.0125 .20172 -50.6875 -6.6875 ]
[-0.0125 .20172 0 6.6875 ]
[-0.0125 .31092 -50.6875 -6.6875 ]
[-0.0125 .31092 0 6.6875 ]
[-0.0125 .42013 -50.6875 -6.6875 ]
[-0.0125 .42013 0 6.6875 ]
[-0.0125 .52934 -50.6875 -6.6875 ]
[-0.0125 .52934 0 6.6875 ]
[ -0.0005 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03283 .32453 m
.03283 .33078 L
s
gsave
.03283 .31203 -84.25 -17.375 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding WindowsANSIEncoding def
        currentdict
    end

    newfontname exch definefont pop
} def

63.000 13.625 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.813 13.625 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.438 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.813 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(8) show
105.000 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.2574 .32453 m
.2574 .33078 L
s
gsave
.2574 .31203 -84 -16.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.00001) show
105.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.48197 .32453 m
.48197 .33078 L
s
gsave
.48197 .31203 -75 -16.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.01) show
87.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.70655 .32453 m
.70655 .33078 L
s
gsave
.70655 .31203 -69 -16.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
75.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.93112 .32453 m
.93112 .33078 L
s
gsave
.93112 .31203 -78 -16.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10000) show
93.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.18623 .32453 m
.18623 .32828 L
s
.20862 .32453 m
.20862 .32828 L
s
.22175 .32453 m
.22175 .32828 L
s
.23108 .32453 m
.23108 .32828 L
s
.23832 .32453 m
.23832 .32828 L
s
.24423 .32453 m
.24423 .32828 L
s
.24924 .32453 m
.24924 .32828 L
s
.25357 .32453 m
.25357 .32828 L
s
.4108 .32453 m
.4108 .32828 L
s
.43319 .32453 m
.43319 .32828 L
s
.44632 .32453 m
.44632 .32828 L
s
.45565 .32453 m
.45565 .32828 L
s
.46289 .32453 m
.46289 .32828 L
s
.46881 .32453 m
.46881 .32828 L
s
.47381 .32453 m
.47381 .32828 L
s
.47815 .32453 m
.47815 .32828 L
s
.63537 .32453 m
.63537 .32828 L
s
.65776 .32453 m
.65776 .32828 L
s
.6709 .32453 m
.6709 .32828 L
s
.68022 .32453 m
.68022 .32828 L
s
.68746 .32453 m
.68746 .32828 L
s
.69338 .32453 m
.69338 .32828 L
s
.69839 .32453 m
.69839 .32828 L
s
.70272 .32453 m
.70272 .32828 L
s
.85995 .32453 m
.85995 .32828 L
s
.88234 .32453 m
.88234 .32828 L
s
.89547 .32453 m
.89547 .32828 L
s
.9048 .32453 m
.9048 .32828 L
s
.91204 .32453 m
.91204 .32828 L
s
.91796 .32453 m
.91796 .32828 L
s
.92296 .32453 m
.92296 .32828 L
s
.9273 .32453 m
.9273 .32828 L
s
.25 Mabswid
0 .32453 m
1 .32453 L
s
0 .09251 m
.00625 .09251 L
s
gsave
-0.0125 .09251 -111.688 -10.6875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding WindowsANSIEncoding def
        currentdict
    end

    newfontname exch definefont pop
} def

63.000 13.625 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.813 13.625 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.438 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.813 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(24) show
109.188 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .20172 m
.00625 .20172 L
s
gsave
-0.0125 .20172 -111.688 -10.6875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding WindowsANSIEncoding def
        currentdict
    end

    newfontname exch definefont pop
} def

63.000 13.625 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.813 13.625 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.438 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.813 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(21) show
109.188 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .31092 m
.00625 .31092 L
s
gsave
-0.0125 .31092 -111.688 -10.6875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding WindowsANSIEncoding def
        currentdict
    end

    newfontname exch definefont pop
} def

63.000 13.625 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.813 13.625 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.438 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.813 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(18) show
109.188 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .42013 m
.00625 .42013 L
s
gsave
-0.0125 .42013 -111.688 -10.6875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding WindowsANSIEncoding def
        currentdict
    end

    newfontname exch definefont pop
} def

63.000 13.625 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.813 13.625 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.438 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.813 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(15) show
109.188 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .52934 m
.00625 .52934 L
s
gsave
-0.0125 .52934 -111.688 -10.6875 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding WindowsANSIEncoding def
        currentdict
    end

    newfontname exch definefont pop
} def

63.000 13.625 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.813 13.625 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.625 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.438 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.813 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(12) show
109.188 9.438 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .0579 m
.00375 .0579 L
s
0 .06878 m
.00375 .06878 L
s
0 .07517 m
.00375 .07517 L
s
0 .07971 m
.00375 .07971 L
s
0 .08323 m
.00375 .08323 L
s
0 .0861 m
.00375 .0861 L
s
0 .08854 m
.00375 .08854 L
s
0 .09065 m
.00375 .09065 L
s
0 .1671 m
.00375 .1671 L
s
0 .17799 m
.00375 .17799 L
s
0 .18438 m
.00375 .18438 L
s
0 .18891 m
.00375 .18891 L
s
0 .19244 m
.00375 .19244 L
s
0 .19531 m
.00375 .19531 L
s
0 .19775 m
.00375 .19775 L
s
0 .19986 m
.00375 .19986 L
s
0 .27631 m
.00375 .27631 L
s
0 .2872 m
.00375 .2872 L
s
0 .29359 m
.00375 .29359 L
s
0 .29812 m
.00375 .29812 L
s
0 .30164 m
.00375 .30164 L
s
0 .30452 m
.00375 .30452 L
s
0 .30696 m
.00375 .30696 L
s
0 .30906 m
.00375 .30906 L
s
0 .38552 m
.00375 .38552 L
s
0 .39641 m
.00375 .39641 L
s
0 .4028 m
.00375 .4028 L
s
0 .40733 m
.00375 .40733 L
s
0 .41085 m
.00375 .41085 L
s
0 .41373 m
.00375 .41373 L
s
0 .41616 m
.00375 .41616 L
s
0 .41827 m
.00375 .41827 L
s
0 .49473 m
.00375 .49473 L
s
0 .50562 m
.00375 .50562 L
s
0 .512 m
.00375 .512 L
s
0 .51654 m
.00375 .51654 L
s
0 .52006 m
.00375 .52006 L
s
0 .52294 m
.00375 .52294 L
s
0 .52537 m
.00375 .52537 L
s
0 .52748 m
.00375 .52748 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.008 w
.02381 .35427 Mdot
.04634 .36527 Mdot
.05953 .37173 Mdot
.06888 .37632 Mdot
.07613 .3799 Mdot
.08707 .38532 Mdot
.09524 .3894 Mdot
.13328 .39228 Mdot
.15033 .39738 Mdot
.20318 .39732 Mdot
.22235 .39802 Mdot
.2775 .39503 Mdot
.29691 .39265 Mdot
.33597 .3851 Mdot
.3532 .34345 Mdot
.36441 .35126 Mdot
.37892 .37379 Mdot
.38891 .38979 Mdot
.39655 .39598 Mdot
.40272 .39465 Mdot
.40881 .4002 Mdot
.41393 .40072 Mdot
.41835 .39995 Mdot
.42225 .402 Mdot
.42573 .39952 Mdot
.42975 .40002 Mdot
.43333 .40335 Mdot
.43656 .40543 Mdot
.43949 .40697 Mdot
.44247 .40439 Mdot
.44519 .40509 Mdot
.44771 .40184 Mdot
.45007 .40312 Mdot
.45227 .39961 Mdot
.45434 .39961 Mdot
.45627 .39998 Mdot
.4581 .39868 Mdot
.45984 .39322 Mdot
.46148 .393 Mdot
.46304 .39714 Mdot
.47023 .39614 Mdot
.47612 .38989 Mdot
.4811 .38372 Mdot
.49232 .39391 Mdot
.50065 .39915 Mdot
.50727 .3931 Mdot
.52287 .39292 Mdot
.53337 .39582 Mdot
.54129 .39434 Mdot
.55091 .38544 Mdot
.55833 .38475 Mdot
.56437 .38702 Mdot
.56945 .38789 Mdot
.57555 .38978 Mdot
.58068 .39558 Mdot
.58511 .39409 Mdot
.58901 .39465 Mdot
.59714 .39634 Mdot
.60363 .39395 Mdot
.60905 .3868 Mdot
.61368 .37795 Mdot
.61774 .34796 Mdot
.62204 .38383 Mdot
.62583 .38768 Mdot
.62922 .39077 Mdot
.6323 .38538 Mdot
.63511 .38968 Mdot
.63917 .38538 Mdot
.64277 .38538 Mdot
.64602 .37442 Mdot
.64897 .37442 Mdot
.65168 .37795 Mdot
.65418 .3821 Mdot
.65748 .37946 Mdot
.66047 .36988 Mdot
.66321 .36988 Mdot
.66574 .35251 Mdot
.66809 .35251 Mdot
.67027 .35251 Mdot
.67232 .36988 Mdot
.67425 .36347 Mdot
.67607 .36988 Mdot
.6778 .37795 Mdot
.67943 .38538 Mdot
.68099 .39042 Mdot
.6827 .38724 Mdot
.68432 .38538 Mdot
.68586 .36347 Mdot
.68733 .36347 Mdot
.68874 .36988 Mdot
.6901 .35251 Mdot
.69157 .36347 Mdot
.69298 .36988 Mdot
.69432 .36347 Mdot
.69562 .36988 Mdot
.69687 .37795 Mdot
.69807 .38538 Mdot
.69937 .38327 Mdot
.70063 .37442 Mdot
.70184 .36347 Mdot
.703 .37795 Mdot
.70413 .38327 Mdot
.70522 .38083 Mdot
.70641 .37442 Mdot
.70756 .35251 Mdot
.70975 .36988 Mdot
.71078 .35251 Mdot
.71277 .35251 Mdot
.71372 .36988 Mdot
.71464 .38538 Mdot
.71554 .39179 Mdot
.71641 .39423 Mdot
.71726 .39423 Mdot
.71824 .39634 Mdot
.71918 .3982 Mdot
.72011 .39906 Mdot
.721 .39987 Mdot
.72187 .39906 Mdot
.72272 .4034 Mdot
.72377 .40064 Mdot
.72479 .3973 Mdot
.72577 .3982 Mdot
.72673 .39423 Mdot
.72766 .39634 Mdot
.72856 .40275 Mdot
.72973 .40138 Mdot
.73086 .40574 Mdot
.73195 .40628 Mdot
.733 .4068 Mdot
.73402 .40402 Mdot
.73501 .40275 Mdot
.73617 .4034 Mdot
.73729 .40519 Mdot
.73837 .40628 Mdot
.73942 .40628 Mdot
.74043 .4073 Mdot
.74142 .40519 Mdot
.74265 .40402 Mdot
.74384 .39906 Mdot
.74498 .39423 Mdot
.74609 .39423 Mdot
.74716 .39906 Mdot
.74819 .39532 Mdot
.7496 .39634 Mdot
.75095 .39042 Mdot
.75224 .39042 Mdot
.75349 .39634 Mdot
.75468 .39532 Mdot
.75584 .3982 Mdot
.75759 .40208 Mdot
.75925 .3973 Mdot
.76082 .3982 Mdot
.76233 .40275 Mdot
.76377 .4034 Mdot
.76515 .40519 Mdot
.76735 .40519 Mdot
.76942 .4068 Mdot
.77136 .40826 Mdot
.7732 .4073 Mdot
.77493 .40872 Mdot
.77825 .40779 Mdot
.78125 .41043 Mdot
.784 .41122 Mdot
.78654 .41269 Mdot
.79472 .41233 Mdot
.80125 .41304 Mdot
.80669 .41269 Mdot
.81134 .41269 Mdot
.83131 .39987 Mdot
.84359 .40402 Mdot
.85249 .36347 Mdot
.89221 .43643 Mdot
.90956 .46349 Mdot
.92824 .46785 Mdot
.94003 .46652 Mdot
.94866 .47738 Mdot
.95548 .47838 Mdot
.96111 .47737 Mdot
.96591 .47772 Mdot
.97009 .47666 Mdot
.97328 .46899 Mdot
.97619 .46997 Mdot
0 0 1 r
.30786 .37819 Mdot
.33039 .38058 Mdot
.34247 .37663 Mdot
.35001 .36728 Mdot
.35552 .37487 Mdot
.35979 .30396 Mdot
.36321 .36354 Mdot
.366 .3767 Mdot
.36858 .38547 Mdot
.37073 .37777 Mdot
.37275 .37848 Mdot
.37447 .36157 Mdot
.3761 .38824 Mdot
.37765 .36996 Mdot
.38055 .35716 Mdot
.38321 .36971 Mdot
.38567 .33641 Mdot
.38796 .34962 Mdot
.3921 .37999 Mdot
.39577 .39202 Mdot
.39907 .4044 Mdot
.40207 .41477 Mdot
.40481 .42334 Mdot
.40734 .40838 Mdot
.40969 .43745 Mdot
.41188 .44307 Mdot
.41586 .36304 Mdot
.4194 .36111 Mdot
.4226 .37541 Mdot
.42551 .37812 Mdot
.42818 .37581 Mdot
.43155 .3787 Mdot
.43461 .38205 Mdot
.4374 .38731 Mdot
.43997 .38487 Mdot
.44235 .37094 Mdot
.44457 .39569 Mdot
.44665 .37721 Mdot
.44861 .38559 Mdot
.45219 .39794 Mdot
.45542 .40265 Mdot
.45836 .36919 Mdot
.46106 .36857 Mdot
.46585 .34804 Mdot
.47004 .37235 Mdot
.47706 .37546 Mdot
.48284 .36549 Mdot
.492 .36277 Mdot
.50499 .3743 Mdot
.51425 .35175 Mdot
.52734 .337 Mdot
.53664 .38494 Mdot
.54387 .36639 Mdot
.54977 .37287 Mdot
.5591 .39512 Mdot
.56634 .39252 Mdot
.57226 .41202 Mdot
.57726 .42046 Mdot
.5816 .43237 Mdot
.58542 .4169 Mdot
.59016 .43111 Mdot
.59429 .45137 Mdot
.59795 .45262 Mdot
.60125 .4563 Mdot
.60424 .45262 Mdot
.60698 .46503 Mdot
.6095 .44778 Mdot
.61184 .42237 Mdot
.61403 .44235 Mdot
.61608 .43867 Mdot
.618 .43337 Mdot
.61982 .42442 Mdot
.62318 .4392 Mdot
.62622 .47 Mdot
.62901 .4684 Mdot
.63157 .4831 Mdot
.63395 .49564 Mdot
.63616 .50734 Mdot
.64018 .37629 Mdot
.64291 .5037 Mdot
.64543 .52028 Mdot
.64776 .51729 Mdot
.64994 .52316 Mdot
.65198 .50401 Mdot
.65391 .53006 Mdot
.65572 .5083 Mdot
.65744 .53735 Mdot
.65907 .54381 Mdot
.66063 .54875 Mdot
.66211 .56095 Mdot
.66353 .5525 Mdot
.66489 .57645 Mdot
.66619 .4723 Mdot
.66744 .59021 Mdot
.66865 .5795 Mdot
.66982 .60206 Mdot
.67094 .60332 Mdot
.67203 .58494 Mdot
.67298 .37795 Mdot
.6739 .36347 Mdot
.67479 .36988 Mdot
.67566 .37795 Mdot
.67651 .36347 Mdot
.67734 .36347 Mdot
.67815 .36988 Mdot
.68045 .37795 Mdot
.68119 .37442 Mdot
.6819 .35251 Mdot
.68329 .36347 Mdot
.6845 .37442 Mdot
.68565 .41436 Mdot
.68677 .35251 Mdot
.68785 .41122 Mdot
.6889 .41002 Mdot
.69028 .39179 Mdot
.69161 .45104 Mdot
.69288 .41615 Mdot
.6941 .46504 Mdot
.69528 .44711 Mdot
.69642 .4773 Mdot
.69752 .47484 Mdot
.69859 .48776 Mdot
.69962 .49488 Mdot
.70062 .49512 Mdot
.70159 .51171 Mdot
.70253 .49412 Mdot
.70344 .52653 Mdot
.70433 .50097 Mdot
.7052 .5396 Mdot
.70604 .53427 Mdot
.70686 .52427 Mdot
.70767 .56365 Mdot
.70845 .53803 Mdot
.70902 .36347 Mdot
.70958 .35251 Mdot
.71012 .37442 Mdot
.71066 .37442 Mdot
.7112 .38327 Mdot
.71172 .38083 Mdot
.71223 .35251 Mdot
.71274 .36988 Mdot
.71324 .35251 Mdot
.71373 .38724 Mdot
.7149 .39042 Mdot
.71558 .42483 Mdot
.71624 .41557 Mdot
.71689 .43635 Mdot
.71753 .39532 Mdot
.71816 .44747 Mdot
.71877 .44225 Mdot
.71937 .45628 Mdot
.71996 .46312 Mdot
.72054 .45939 Mdot
.72111 .47867 Mdot
.72167 .41338 Mdot
.72222 .49118 Mdot
.72276 .48351 Mdot
.72329 .50045 Mdot
.72382 .50554 Mdot
.72433 .50476 Mdot
.72484 .47426 Mdot
.72534 .4893 Mdot
.72583 .50178 Mdot
.72632 .49581 Mdot
.7268 .49764 Mdot
.72773 .51671 Mdot
.72855 .49343 Mdot
.72935 .52902 Mdot
.73013 .51311 Mdot
.73089 .54007 Mdot
.73163 .54691 Mdot
.73236 .55888 Mdot
.73307 .54198 Mdot
.73377 .54718 Mdot
.73445 .57504 Mdot
.73512 .54321 Mdot
.73577 .54966 Mdot
.73641 .5233 Mdot
.73704 .55594 Mdot
.73766 .52212 Mdot
.73827 .56166 Mdot
.73886 .57466 Mdot
.73944 .58577 Mdot
.74002 .57317 Mdot
.74058 .59251 Mdot
.74113 .57329 Mdot
.74168 .54774 Mdot
.74221 .58211 Mdot
.74274 .58266 Mdot
.74326 .55075 Mdot
.74377 .38083 Mdot
.74427 .38327 Mdot
.74477 .36988 Mdot
.74526 .37442 Mdot
.74621 .37442 Mdot
.74714 .37442 Mdot
.74901 .35251 Mdot
.74986 .35251 Mdot
.7507 .37442 Mdot
.75151 .36347 Mdot
.75307 .36347 Mdot
.75456 .35251 Mdot
.75598 .36988 Mdot
.75735 .35251 Mdot
.75992 .36988 Mdot
.76452 .37795 Mdot
.77213 .37442 Mdot
.78348 .35251 Mdot
.79854 .36347 Mdot
.81657 .38083 Mdot
.8366 .37795 Mdot
.87965 .38724 Mdot
.90184 .35251 Mdot
.92412 .35251 Mdot
.93722 .38327 Mdot
.94653 .35251 Mdot
.95376 .35251 Mdot
.95967 .35251 Mdot
.969 .35251 Mdot
.97282 .35251 Mdot
.97619 .35251 Mdot
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[19]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00
=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool0
0`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000
oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?oo
o`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?oo
o`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3o
ool00`000000oooo0?ooo`3X0?ooo`00=@3oool200000>T0oooo000e0?ooo`030000003oool0oooo
0>P0oooo000e0?ooo`800000j@3oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo0P00
003Y0?ooo`00=@3oool200000>T0oooo00030?ooo`D000000P3oool2000000@0oooo00D000000?oo
o`3oool0oooo000000030?ooo`D000000P3oool2000001@0oooo0P00003Y0?ooo`001@3oool00`00
0000oooo0?ooo`090?ooo`030000003oool0000000H0oooo00D000000?ooo`3oool0oooo00000002
0?ooo`030000003oool0oooo0140oooo0P00003Y0?ooo`001@3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0
oooo0140oooo0`00003X0?ooo`001@3oool00`000000oooo0?ooo`090?ooo`030000003oool00000
00H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo0140oooo00<00000
0?ooo`3oool0j03oool000D0oooo00<000000?ooo`3oool0203oool01@000000oooo0?ooo`3oool0
000000D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00@0oooo0P00
00030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0j03oool000@0oooo0P00000C
0?ooo`800000103oool2000000<0oooo0`0000000`3oool000000?ooo`020?ooo`8000001`3oool0
0`000000oooo0?ooo`3X0?ooo`00:@3oool00`000000oooo0?ooo`02000000L0oooo00<000000?oo
o`3oool0j03oool002P0oooo0P0000030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03o
ool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0
oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<0
00000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?oo
o`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0
j03oool003D0oooo0P00003Y0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`00
0000oooo0?ooo`3X0?ooo`00=@3oool200000>T0oooo000e0?ooo`800000j@3oool003D0oooo0P00
003Y0?ooo`000`3oool500000080oooo0P0000040?ooo`050000003oool0oooo0?ooo`0000000`3o
ool500000080oooo0P00000D0?ooo`800000j@3oool000D0oooo00<000000?ooo`3oool02@3oool0
0`000000oooo000000060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`0A0?ooo`800000j@3oool000D0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`05
0?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0A0?ooo`<00000j03o
ool000D0oooo00<000000?ooo`3oool02@3oool00`000000oooo000000060?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo0>P0oooo0005
0?ooo`030000003oool0oooo00P0oooo00D000000?ooo`3oool0oooo000000050?ooo`050000003o
ool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`040?ooo`8000000P3oool2000000L0oooo
00<000000?ooo`3oool0j03oool000@0oooo0P00000C0?ooo`800000103oool2000000<0oooo0`00
00000`3oool000000?ooo`030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0j03o
ool002T0oooo00D000000?ooo`3oool0oooo000000070?ooo`030000003oool0oooo0>P0oooo000X
0?ooo`8000000P3oool2000000L0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?oo
o`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0
j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool0
03D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo
00<000000?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool003D0oooo00<00000
0?ooo`3oool0j03oool003D0oooo00<000000?ooo`3oool0j03oool002T0oooo1@0000020?ooo`80
00000`3oool2000000<0oooo00@000000?ooo`3oool0oooo1@0000020?ooo`800000f@3oool002/0
oooo00<000000?ooo`3oool01`3oool01@000000oooo0000003oool0000000H0oooo00D000000?oo
o`3oool0oooo000000020?ooo`030000003oool0oooo01@0oooo0P0000040?ooo`800000103oool2
000000@0oooo0P0000040?ooo`800000103oool2000000<0oooo1@00000E0?ooo`800000103oool2
000000@0oooo0P0000030?ooo`D000008`3oool500000080oooo0P00000S0?ooo`D000000P3oool2
000000@0oooo0P0000040?ooo`800000103oool2000000D0oooo000[0?ooo`030000003oool0oooo
00L0oooo00@000000?ooo`3oool000001`3oool01@000000oooo0?ooo`3oool000000080oooo00<0
00000?ooo`3oool04`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0
oooo000000@0oooo00<000000?ooo`3oool0503oool010000000oooo0?ooo`0000080?ooo`040000
003oool0oooo000000@0oooo00<000000?ooo`3oool09@3oool01@000000oooo0?ooo`3oool00000
0080oooo00<000000?ooo`3oool08P3oool01@000000oooo0?ooo`3oool000000080oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080
oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01`3oool3000000030?oo
o`000000oooo00D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo01<0
oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0
oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`03
0000003oool0oooo01@0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`000004
0?ooo`030000003oool0oooo02D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003o
ool0oooo0280oooo00D000000?ooo`3oool0oooo000000020?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0
oooo0080oooo000[0?ooo`030000003oool0oooo00L0oooo0P0000030?ooo`030000003oool0oooo
00<0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00@0oooo0P00000=
0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo
0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool0
0`000000oooo0?ooo`0D0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool00000
103oool00`000000oooo0?ooo`0U0?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000
oooo0?ooo`0R0?ooo`050000003oool0oooo0?ooo`0000000P3oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool00`000000oooo
0?ooo`020?ooo`00:P3oool2000000T0oooo0P0000080?ooo`800000103oool2000000<0oooo0`00
00000`3oool000000000000=0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@00000
0?ooo`3oool00000103oool00`000000oooo0?ooo`0D0?ooo`040000003oool0oooo000000P0oooo
00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`0U0?ooo`050000003oool0oooo0?oo
o`0000000P3oool00`000000oooo0?ooo`0R0?ooo`050000003oool0oooo0?ooo`0000000P3oool0
10000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000P3oool00`000000oooo0?ooo`020?ooo`00=@3oool2000001L0oooo0P00000>0?ooo`800000
2P3oool2000000@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`030000
o`3oool0oooo0100oooo0P00000:0?ooo`800000103oool2000002H0oooo0P0000040?ooo`800000
903oool2000000@0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000050?ooo`000`3o
ool500000080oooo0P0000040?ooo`050000003oool0oooo0?ooo`0000000`3oool500000080oooo
0P00000D0?ooo`8000005`3oool200000=00oooo00050?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`0000001P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0
4@3oool300000>P0oooo00050?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool01@3o
ool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool04@3oool00`000000oooo
0?ooo`3X0?ooo`001@3oool00`000000oooo0?ooo`090?ooo`030000003oool0000000H0oooo00D0
00000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0
j03oool000D0oooo00<000000?ooo`3oool0203oool01@000000oooo0?ooo`3oool0000000D0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00@0oooo0P0000020?ooo`80
00001`3oool00`000000oooo0?ooo`3X0?ooo`00103oool2000001<0oooo0P0000040?ooo`800000
0`3oool300000080oooo00<000000?ooo`3oool00P0000070?ooonX000000@3oool002T0oooo00<0
00000?ooo`3oool00P0000070?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0803o
ool00`000000oooo0?ooo`030?ooo`050000003oool0oooo0000003oool00P0000000`3oool00000
00000002000002<0oooo00<000000?ooo`3oool00P3oool01`000000oooo0?ooo`000000oooo0000
003oool01@00000S0?ooo`030000003oool0oooo0080oooo00L000000?ooo`3oool000000?ooo`00
0000oooo00D000008`3oool00`000000oooo0?ooo`020?ooo`070000003oool0oooo0000003oool0
00000?ooo`02000000040?ooo`000000000000000140oooo000X0?ooo`8000000P3oool2000000L0
oooo00<000000?ooo`3oool0EP3oool00`000?l0oooo0?ooo`0N0?ooo`030000o`3oool0oooo06h0
oooo000e0?ooo`030000003oool0oooo04l0oooo00<000000?ooo`3oool0UP3oool003D0oooo00<0
00000?ooo`3oool0J@3oool00`000?l0oooo0?ooo`0P0?ooo`030000003oool0oooo05T0oooo000e
0?ooo`030000003oool0oooo0540oooo00<000000?ooo`3oool00`3oool00`000?l0oooo0?ooo`0J
0?ooo`030000o`3oool0oooo0740oooo000e0?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool0TP3oool3000000040?ooo`000?l0003o000000<0oooo00<000000000o`3oool01`3oool3
0000o`D0oooo00<0003o0?ooo`3oool0603oool00`000?l0oooo0?ooo`020?ooo`030000o`3oool0
oooo00<0oooo0P000?l00`3oool0003o0?ooo`030000o`L0oooo000e0?ooo`030000003oool0oooo
05D0oooo00<0003o0?ooo`3oool0T03oool003D0oooo00<000000?ooo`3oool01`3oool00`000000
oooo0?ooo`170?ooo`040000o`3oool0oooo0000o`P0oooo0P000?l@0?ooo`030000o`3oool0oooo
02L0oooo0P000?l0203oool0003o000000000000oooo0000003oool0003o203oool20000o`T0oooo
00<0003o0?ooo`3oool02P3oool00`000000oooo0?ooo`0R0?ooo`00=@3oool00`000000oooo0?oo
o`1>0?ooo`030000o`3oool0oooo01H0oooo0P000?l40?ooo`030000o`3oool0oooo00/0oooo00<0
003o0?ooo`3oool0JP3oool003D0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`16
0?ooo`030000o`000000003o00d0oooo00<0003o0?ooo`3oool00`3oool00`000?l0oooo0?ooo`0?
0?ooo`030000o`3oool0oooo01L0oooo0P0000000`3oool000000000o`020?ooo`8000000`3oool0
0`000?l000000?ooo`050?ooo`040000o`3oool0003o0000ocX0oooo000e0?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool0=03oool00`000?l0oooo0?ooo`050?ooo`040000o`3oool0
oooo0000o`80oooo0P000?l;0?ooo`80003o103oool00`000?l0oooo0?ooo`040?ooo`030000o`3o
ool0oooo00<0oooo00<0003o0?ooo`3oool05P3oool00`000000oooo0?ooo`040?ooo`030000o`00
0000000000@0oooo10000?l20?ooo`<0000000<0003o0?ooo`3oool01@3oool20000o`80oooo00<0
003o0?ooo`000?l03P3oool00`000?l0oooo0?ooo`0V0?ooo`00=@3oool00`000000oooo0?ooo`0>
0?ooo`030000003oool0oooo03T0oooo00<0003o0?ooo`3oool01P3oool00`000?l0oooo0?ooo`02
0?ooo`030000o`3oool0oooo00H0oooo0P000?lb0?ooo`8000002P3oool00`000000003o0?ooo`06
0?ooo`030000o`3oool0oooo00h0oooo00<0003o0?ooo`3oool0:P3oool003D0oooo00<000000?oo
o`3oool04@3oool00`000000oooo0?ooo`0g0?ooo`030000003oool0oooo00@0oooo00<0003o0?oo
o`3oool03P3oool00`000?l0oooo0000o`070?ooo`030000003oool0oooo00T0oooo00@0003o0?oo
o`3oool0oooo0P00000;0?ooo`040000003oool0oooo00000080oooo0`0000080?ooo`8000000P3o
ool2000000030?ooo`000?l0000000H0oooo00<0003o0?ooo`3oool0:P3oool00`000?l0oooo0?oo
o`0>0?ooo`00=@3oool2000001@0oooo00<000000?ooo`3oool0?P3oool010000?l0oooo0?ooo`00
000:0?ooo`030000o`3oool0oooo00H0oooo00<000000?ooo`3oool04P3oool3000000/0oooo0`00
000:0?ooo`030000003oool0oooo00D0oooo00<0003o0?ooo`3oool01@3oool2000001`0oooo00<0
003o0?ooo`3oool0703oool003D0oooo00<000000?ooo`3oool0703oool00`000000oooo0?ooo`0N
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0503oool00`000?l000000?ooo`0<
0?ooo`030000003oool0oooo00@0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003o
ool0oooo0080oooo00@000000?ooo`3oool0oooo0P000?l40?ooo`8000000P3oool00`000000oooo
0?ooo`0A0?ooo`030000o`3oool0oooo00<0oooo00@0003o0?ooo`3oool000000`3oool3000003/0
oooo000e0?ooo`8000008@3oool00`000000oooo0?ooo`090?ooo`050000003oool0oooo0?ooo`00
0000:03oool00`000000oooo0?ooo`040?ooo`070000003oool0oooo0?ooo`000?l0oooo0000o`02
000000030?ooo`000000oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`08
0?ooo`050000003oool0oooo0?ooo`0000006`3oool3000000<0oooo1@00000i0?ooo`00=@3oool0
0`000000oooo0?ooo`1L0?ooo`@0000000<0oooo0000003oool00P3oool00`000000oooo0000o`0l
0?ooo`H000000`3oool2000000l0oooo00@000000?ooo`3oool000009P3oool003D0oooo0P00001J
0?ooo`030000o`3oool0003o00H0oooo0`0000120?ooo`<000001@3oool2000000030?ooo`000000
oooo03@0oooo000e0?ooo`800000P`3oool00`000?l0oooo0?ooo`0H0?ooo`030000o`3oool0oooo
0140oooo1@0000020?ooo`800000;P3oool000<0oooo1@0000020?ooo`800000103oool01@000000
oooo0?ooo`3oool0000000<0oooo1@0000020?ooo`800000503oool2000005/0oooo00<0003o0?oo
o`3oool0:03oool00`000?l0oooo0?ooo`0D0?ooo`030000o`3oool0003o00@0oooo0P000?lB0?oo
o`030000003oool0oooo02h0oooo00050?ooo`030000003oool0oooo00T0oooo00<000000?ooo`00
00001P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool04@3oool20000
08@0oooo00<0003o0?ooo`3oool0HP3oool000D0oooo00<000000?ooo`3oool02P3oool00`000000
oooo0?ooo`050?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0A0?oo
o`<00000F`3oool00`000?l0oooo0?ooo`0]0?ooo`030000o`3oool0003o01D0oooo00<0003o0?oo
o`3oool0@P3oool000D0oooo00<000000?ooo`3oool02@3oool00`000000oooo000000060?ooo`05
0000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo
0>P0oooo00050?ooo`030000003oool0oooo00P0oooo00D000000?ooo`3oool0oooo000000050?oo
o`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`040?ooo`8000000P3oool3
000000H0oooo00<000000?ooo`3oool0Q03oool00`000?l0oooo0000o`050?ooo`030000o`3oool0
oooo05T0oooo00040?ooo`8000004`3oool2000000@0oooo0P0000030?ooo`<000000P3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0G03oool00`00
0?l0oooo0?ooo`140?ooo`030000o`3oool0oooo02H0oooo00<000000?ooo`3oool06@3oool002T0
oooo00<000000?ooo`3oool00P0000070?ooo`030000003oool0oooo08/0oooo0P000?l00`3oool0
003o0?ooo`0D0?ooo`030000o`3oool0oooo0440oooo000X0?ooo`8000000P3oool2000000L0oooo
00<000000?ooo`3oool0G03oool00`000?l0oooo0?ooo`0o0?ooo`030000o`3oool0oooo04L0oooo
000e0?ooo`030000003oool0oooo08L0oooo00@0003o0?ooo`3oool0003o4P3oool00`000?l0oooo
0?ooo`040?ooo`030000o`3oool0oooo0440oooo000e0?ooo`030000003oool0oooo08P0oooo0P00
0?mN0?ooo`00=@3oool00`000000oooo0?ooo`280?ooo`030000o`3oool0oooo01T0oooo00<0003o
0?ooo`3oool0@@3oool003D0oooo00<000000?ooo`3oool0Y03oool00`000?l0oooo0?ooo`0Y0?oo
o`030000003oool0oooo01D0oooo000e0?ooo`030000003oool0oooo08X0oooo00<0003o0?ooo`3o
ool00P3oool00`000?l0oooo0?ooo`0<0?ooo`030000o`3oool0oooo03<0oooo00@000000?ooo`3o
ool00000403oool003D0oooo00<000000?ooo`3oool0SP3oool00`000?l0oooo0?ooo`070?ooo`03
0000o`3oool0oooo04@0oooo0P0000070?ooo`00=@3oool00`000000oooo0?ooo`2O0?ooo`030000
o`3oool0oooo00<0oooo00<0003o0?ooo`3oool0=P3oool00`000000oooo0?ooo`070?ooo`00=@3o
ool00`000000oooo0?ooo`2O0?ooo`030000o`3oool0oooo0080oooo00<0003o0?ooo`3oool0<P3o
ool00`000000oooo00000002000000X0oooo000e0?ooo`030000003oool0oooo0900oooo00<0003o
0?ooo`3oool04P3oool00`000?l0oooo0?ooo`100?ooo`00=@3oool00`000000oooo0?ooo`2O0?oo
o`030000o`3oool0oooo00<0oooo00<0003o0?ooo`3oool0@03oool003D0oooo00<000000?ooo`3o
ool0X03oool00`000?l0oooo0?ooo`020?ooo`80003o@@3oool003D0oooo0P00002A0?ooo`030000
o`3oool0oooo00`0oooo0P000?l50?ooo`030000o`3oool0oooo03l0oooo000e0?ooo`030000003o
ool0oooo0:00oooo00<0003o0?ooo`3oool00P3oool00`000?l0oooo0?ooo`100?ooo`00=@3oool0
0`000000oooo0?ooo`2A0?ooo`80003o00<0oooo0000o`3oool03`3oool00`000?l0oooo0?ooo`10
0?ooo`00=@3oool2000009H0oooo00<0003o0?ooo`3oool0D03oool003D0oooo0P00002Q0?ooo`03
0000o`3oool0oooo00<0oooo00<0003o0?ooo`3oool0?`3oool003D0oooo0P00002D0?ooo`030000
o`3oool0oooo0100oooo00<0003o0?ooo`3oool0?`3oool000<0oooo1@0000020?ooo`800000103o
ool01@000000oooo0?ooo`3oool0000000<0oooo1@0000020?ooo`800000503oool2000009@0oooo
0P000?l<0?ooo`030000o`3oool0oooo00@0oooo00<0003o0?ooo`3oool0?@3oool000D0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo000000060?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo0?ooo`0A0?ooo`800000X@3oool00`000?l0oooo0?ooo`150?ooo`001@3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00D0oooo00D000000?ooo`3oool0oooo
000000020?ooo`030000003oool0oooo0140oooo0`00002E0?ooo`030000o`3oool0oooo00h0oooo
00<0003o0?ooo`3oool0?`3oool000D0oooo00<000000?ooo`3oool02@3oool00`000000oooo0000
00060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0A0?ooo`030000
003oool0oooo09H0oooo00<0003o0?ooo`3oool0203oool00`000?l0oooo0?ooo`140?ooo`001@3o
ool00`000000oooo0?ooo`080?ooo`050000003oool0oooo0?ooo`0000001@3oool01@000000oooo
0?ooo`3oool000000080oooo00<000000?ooo`3oool0103oool200000080oooo0P0000070?ooo`03
0000003oool0oooo0:40oooo00<0003o0?ooo`3oool00`3oool00`000?l0oooo0?ooo`0n0?ooo`00
103oool2000001<0oooo0P0000040?ooo`8000000`3oool300000080oooo00@000000?ooo`3oool0
0000203oool00`000000oooo0?ooo`2F0?ooo`030000o`3oool0oooo00h0oooo0P000?lo0?ooo`00
:@3oool01@000000oooo0?ooo`3oool0000000L0oooo00<000000?ooo`3oool0UP3oool00`000?l0
oooo0?ooo`0>0?ooo`<0003o?P3oool002P0oooo0P0000020?ooo`8000001`3oool00`000000oooo
0?ooo`2G0?ooo`030000o`3oool0oooo00l0oooo00<0003o0?ooo`3oool0?03oool003D0oooo00<0
00000?ooo`3oool0Y`3oool20000ocl0oooo000e0?ooo`030000003oool0oooo09L0oooo00<0003o
0?ooo`3oool03P3oool00`000?l0oooo0?ooo`0m0?ooo`00=@3oool00`000000oooo0?ooo`2Q0?oo
o`030000o`3oool0oooo04@0oooo000e0?ooo`030000003oool0oooo0>P0oooo000e0?ooo`030000
003oool0oooo0:L0oooo0`000?ln0?ooo`00=@3oool00`000000oooo0?ooo`2G0?ooo`80003oC`3o
ool003D0oooo00<000000?ooo`3oool0Z@3oool00`000?l0oooo0?ooo`0l0?ooo`00=@3oool00`00
0000oooo0?ooo`2I0?ooo`030000o`3oool0oooo00d0oooo00<0003o0?ooo`3oool0?03oool003D0
oooo00<000000?ooo`3oool0V03oool00`000?l0oooo0?ooo`0>0?ooo`030000o`3oool0oooo03`0
oooo000e0?ooo`030000003oool0oooo0>P0oooo000e0?ooo`030000003oool0oooo0>P0oooo000e
0?ooo`030000003oool0oooo09P0oooo0P000?m>0?ooo`00=@3oool00`000000oooo0?ooo`3X0?oo
o`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3o
ool00`000000oooo0?ooo`3X0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-11.519, -28.5091, \
0.0574778, 0.118196}}]
}, Open  ]],

Cell["\<\
In this case, both solutions satisfied the balance equations well beyond \
expected tolerances. Note that even though the error in the balance equation \
was greater at some points for the DAE solution, over the long term, the DAE \
solution is brought back to better satisfy the constraint once the range of \
quick variation is passed.\
\>", "Text"],

Cell["You may want to solve some DAEs of the form", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{GridBox[{
            {
              RowBox[{
                RowBox[{
                  SuperscriptBox["x", "\[Prime]",
                    MultilineFunction->None], "(", "t", ")"}], 
                "=", \(f(t, x(t))\)}]},
            {\(\(g(t, x(t)) = 0\)\(\ \)\(,\)\)}
            }], "\[NoBreak]"}], TraditionalForm]], "DisplayMath"],

Cell[TextData[{
  "such that the solution of the differential equation is required to satisfy \
a particular constraint. ",
  StyleBox["NDSolve", "MR"],
  " cannot handle such DAEs directly because the index is too high and ",
  StyleBox["NDSolve", "MR"],
  " expects the number of equations to be the same as the number of dependent \
variables. ",
  StyleBox["NDSolve", "MR"],
  " does, however, have a ",
  StyleBox["Projection", "MR"],
  " method that will often solve the problem."
}], "Text"],

Cell["\<\
A very simple example of such a constrained system is a nonlinear oscillator \
modelling the motion of a pendulum.\
\>", "Text"],

Cell["\<\
This defines the equation, invariant constraint, and starting condition for a \
simulation of the motion of a pendulum.\
\>", "MathCaption"],

Cell[BoxData[{
    \(equation\  = \ \(x''\)[t]\  + \ Sin[x[t]]\  \[Equal] \ 
        0; \), "\[IndentingNewLine]", 
    \(invariant\  = \ \(x'\)[t]\^2 - \ 
        2  Cos[x[t]]; \), "\[IndentingNewLine]", 
    \(start\  = \ {x[0]\  \[Equal] \ 1, \ \(x'\)[0]\  \[Equal] \ 
          0}; \)}], "Input",
  CellLabel->"In[23]:="],

Cell["\<\
Note that the differential equation is effectively the derivative of the \
invariant, so one way to solve the equation is to use the invariant.\
\>", "Text"],

Cell[TextData[{
  "This solves for the motion of a pendulum using the invariant equation. The \
",
  StyleBox["SolveDelayed", "MR"],
  " option tells ",
  StyleBox["NDSolve", "MR"],
  " not to symbolically solve the quadratic equation for ",
  StyleBox["x'",
    FontSlant->"Italic"],
  ", but instead to solve the system as a DAE."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(isol\  = \ 
      First[NDSolve[{invariant\  \[Equal] \ \(-2\)\ Cos[1], \ start}, 
          x, {t, 0, 1000}, \ SolveDelayed \[Rule] True]]\)], "Input",
  CellLabel->"In[26]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"x", "\[Rule]", 
        TagBox[\(InterpolatingFunction[{{0.`, 1000.`}}, "<>"]\),
          False,
          Editable->False]}], "}"}]], "Output",
  CellLabel->"Out[26]="]
}, Open  ]],

Cell[TextData[{
  "However, this solution may not be quite what you expect: the invariant \
equation has the solution ",
  StyleBox["x",
    FontSlant->"Italic"],
  "[",
  StyleBox["t",
    FontSlant->"Italic"],
  "] == constant when it starts with ",
  StyleBox["x'",
    FontSlant->"Italic"],
  "[",
  StyleBox["t",
    FontSlant->"Italic"],
  "] == 0. In fact it does not have unique solutions from this starting \
point. This is because if you do actually solve for ",
  StyleBox["x'",
    FontSlant->"Italic"],
  ", the function does not satisfy the continuity requirements for \
uniqueness. "
}], "Text"],

Cell[TextData[{
  "This solves for the motion of a pendulum using only the differential \
equation. The method ",
  StyleBox["ExplicitRungeKutta", "MR"],
  " is used because it can also be a submethod of the projection method."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(dsol\  = \ 
      First[NDSolve[{equation, start}, x, {t, 0, 2000}, \ 
          Method \[Rule] ExplicitRungeKutta]]\)], "Input",
  CellLabel->"In[27]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"x", "\[Rule]", 
        TagBox[\(InterpolatingFunction[{{0.`, 2000.`}}, "<>"]\),
          False,
          Editable->False]}], "}"}]], "Output",
  CellLabel->"Out[27]="]
}, Open  ]],

Cell["\<\
This shows the solution plotted over the last several periods.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot[x[t]\  /. \ dsol, \ {t, 1950, \ 2000}]; \)], "Input",
  CellLabel->"In[28]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-37.1191 0.0190476 0.309019 0.294304 [
[.21429 .29652 -12 -9 ]
[.21429 .29652 12 0 ]
[.40476 .29652 -12 -9 ]
[.40476 .29652 12 0 ]
[.59524 .29652 -12 -9 ]
[.59524 .29652 12 0 ]
[.78571 .29652 -12 -9 ]
[.78571 .29652 12 0 ]
[.97619 .29652 -12 -9 ]
[.97619 .29652 12 0 ]
[.01131 .01472 -12 -4.5 ]
[.01131 .01472 0 4.5 ]
[.01131 .16187 -24 -4.5 ]
[.01131 .16187 0 4.5 ]
[.01131 .45617 -18 -4.5 ]
[.01131 .45617 0 4.5 ]
[.01131 .60332 -6 -4.5 ]
[.01131 .60332 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .30902 m
.21429 .31527 L
s
[(1960)] .21429 .29652 0 1 Mshowa
.40476 .30902 m
.40476 .31527 L
s
[(1970)] .40476 .29652 0 1 Mshowa
.59524 .30902 m
.59524 .31527 L
s
[(1980)] .59524 .29652 0 1 Mshowa
.78571 .30902 m
.78571 .31527 L
s
[(1990)] .78571 .29652 0 1 Mshowa
.97619 .30902 m
.97619 .31527 L
s
[(2000)] .97619 .29652 0 1 Mshowa
.125 Mabswid
.0619 .30902 m
.0619 .31277 L
s
.1 .30902 m
.1 .31277 L
s
.1381 .30902 m
.1381 .31277 L
s
.17619 .30902 m
.17619 .31277 L
s
.25238 .30902 m
.25238 .31277 L
s
.29048 .30902 m
.29048 .31277 L
s
.32857 .30902 m
.32857 .31277 L
s
.36667 .30902 m
.36667 .31277 L
s
.44286 .30902 m
.44286 .31277 L
s
.48095 .30902 m
.48095 .31277 L
s
.51905 .30902 m
.51905 .31277 L
s
.55714 .30902 m
.55714 .31277 L
s
.63333 .30902 m
.63333 .31277 L
s
.67143 .30902 m
.67143 .31277 L
s
.70952 .30902 m
.70952 .31277 L
s
.74762 .30902 m
.74762 .31277 L
s
.82381 .30902 m
.82381 .31277 L
s
.8619 .30902 m
.8619 .31277 L
s
.9 .30902 m
.9 .31277 L
s
.9381 .30902 m
.9381 .31277 L
s
.25 Mabswid
0 .30902 m
1 .30902 L
s
.02381 .01472 m
.03006 .01472 L
s
[(-1)] .01131 .01472 1 0 Mshowa
.02381 .16187 m
.03006 .16187 L
s
[(-0.5)] .01131 .16187 1 0 Mshowa
.02381 .45617 m
.03006 .45617 L
s
[(0.5)] .01131 .45617 1 0 Mshowa
.02381 .60332 m
.03006 .60332 L
s
[(1)] .01131 .60332 1 0 Mshowa
.125 Mabswid
.02381 .04415 m
.02756 .04415 L
s
.02381 .07358 m
.02756 .07358 L
s
.02381 .10301 m
.02756 .10301 L
s
.02381 .13244 m
.02756 .13244 L
s
.02381 .1913 m
.02756 .1913 L
s
.02381 .22073 m
.02756 .22073 L
s
.02381 .25016 m
.02756 .25016 L
s
.02381 .27959 m
.02756 .27959 L
s
.02381 .33845 m
.02756 .33845 L
s
.02381 .36788 m
.02756 .36788 L
s
.02381 .39731 m
.02756 .39731 L
s
.02381 .42674 m
.02756 .42674 L
s
.02381 .4856 m
.02756 .4856 L
s
.02381 .51503 m
.02756 .51503 L
s
.02381 .54446 m
.02756 .54446 L
s
.02381 .57389 m
.02756 .57389 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .59169 m
.02605 .58114 L
.02846 .56625 L
.03279 .53048 L
.03793 .47463 L
.04262 .41383 L
.05293 .26385 L
.06244 .13566 L
.06749 .08247 L
.07006 .06067 L
.07283 .04171 L
.07428 .03374 L
.0756 .02764 L
.07695 .02263 L
.0782 .01907 L
.0789 .01753 L
.07964 .01626 L
.08047 .01529 L
.08122 .01482 L
.08194 .01472 L
.08271 .01502 L
.08344 .01566 L
.0841 .01658 L
.08529 .01894 L
.08656 .02252 L
.08776 .0269 L
.08885 .03172 L
.09153 .04681 L
.09406 .06527 L
.0991 .11307 L
.10458 .1793 L
.12396 .45426 L
.12928 .51583 L
.13206 .54248 L
.135 .56569 L
.13758 .58168 L
.13895 .5884 L
.14041 .59422 L
.14175 .59833 L
.14297 .60101 L
.1442 .60268 L
.14536 .60331 L
.14653 .60301 L
.14763 .60189 L
.14883 .59972 L
.15012 .59629 L
.15147 .59151 L
.15276 .58585 L
.15516 .57233 L
.15949 .53896 L
Mistroke
.1642 .49052 L
.17492 .34684 L
.18463 .20561 L
.18951 .14225 L
.19478 .08545 L
.19721 .06437 L
.19976 .04595 L
.20195 .03348 L
.20309 .02817 L
.20434 .02338 L
.20507 .02105 L
.20575 .01922 L
.20639 .01779 L
.20709 .01652 L
.2083 .01512 L
.20958 .01473 L
.21031 .01501 L
.21098 .01558 L
.21171 .01656 L
.21248 .01798 L
.21378 .02129 L
.2152 .0262 L
.21774 .03831 L
.22045 .05581 L
.22533 .09848 L
.24528 .36762 L
.25578 .50397 L
.26106 .55405 L
.26407 .57519 L
.26541 .5828 L
.26688 .58975 L
.26828 .59509 L
.26957 .59885 L
.27081 .60139 L
.27213 .60294 L
.27331 .60332 L
.27441 .60282 L
.27562 .60131 L
.27693 .59857 L
.27824 .59468 L
.27967 .5891 L
.28256 .57376 L
.28523 .55484 L
.28772 .53335 L
.29699 .42558 L
.30691 .28227 L
.31714 .14216 L
.32291 .0808 L
.32566 .05795 L
.32823 .04074 L
Mistroke
.33047 .0292 L
.33173 .02417 L
.33289 .0205 L
.33414 .01751 L
.33485 .01629 L
.33551 .01547 L
.33672 .01474 L
.33801 .01506 L
.33871 .01571 L
.33936 .0166 L
.34006 .0179 L
.34081 .01965 L
.34207 .02344 L
.34345 .02881 L
.346 .04202 L
.34838 .05809 L
.35339 .10302 L
.35884 .16655 L
.36874 .30653 L
.37832 .44286 L
.38301 .49972 L
.3873 .5425 L
.39183 .57607 L
.39421 .58852 L
.39541 .59337 L
.39671 .59759 L
.39741 .59938 L
.39808 .60076 L
.39931 .60255 L
.40001 .6031 L
.40065 .60331 L
.40135 .60323 L
.40209 .60277 L
.40332 .60118 L
.40447 .59876 L
.40574 .59508 L
.40709 .58994 L
.40979 .57612 L
.41273 .55578 L
.41808 .50572 L
.42811 .37672 L
.43827 .22792 L
.44783 .10732 L
.45313 .05904 L
.45611 .03922 L
.45753 .03179 L
.45885 .02604 L
.46006 .0218 L
.46117 .01879 L
.46239 .01641 L
Mistroke
.4637 .015 L
.46497 .01475 L
.46568 .01509 L
.46634 .01571 L
.46754 .01762 L
.46882 .02072 L
.47003 .02464 L
.47134 .03002 L
.47373 .04274 L
.47823 .07635 L
.48348 .13013 L
.48835 .19148 L
.50665 .45204 L
.51172 .51153 L
.51439 .53772 L
.51721 .561 L
.51968 .57741 L
.52099 .58452 L
.52239 .59091 L
.52368 .59563 L
.52486 .59896 L
.52604 .60137 L
.52716 .60277 L
.52823 .60331 L
.5294 .603 L
.53063 .60168 L
.53179 .59949 L
.53286 .59665 L
.53385 .59336 L
.53607 .58358 L
.53832 .57036 L
.54079 .55225 L
.54576 .50497 L
.55669 .36277 L
.56701 .21225 L
.57627 .09886 L
.57881 .07489 L
.58149 .05357 L
.58379 .0389 L
.58631 .02667 L
.58756 .02219 L
.58822 .02024 L
.58892 .01849 L
.5902 .01614 L
.59139 .01496 L
.59264 .01477 L
.59383 .01557 L
.59489 .0171 L
.59603 .0196 L
.59729 .02339 L
Mistroke
.59868 .02878 L
.60147 .04347 L
.60406 .06152 L
.60648 .082 L
.61559 .18638 L
.62535 .32674 L
.63554 .46804 L
.6413 .53113 L
.64404 .555 L
.64662 .57328 L
.64885 .58574 L
.65 .59089 L
.65126 .59551 L
.65256 .59917 L
.65327 .60069 L
.65393 .60179 L
.65512 .60302 L
.65638 .60329 L
.65708 .60296 L
.65772 .60237 L
.65843 .6014 L
.65918 .59999 L
.66043 .59679 L
.66181 .59204 L
.66436 .57996 L
.66675 .56487 L
.67174 .5221 L
.67717 .46049 L
.68706 .32225 L
.69662 .18496 L
.70131 .12669 L
.70561 .08219 L
.71012 .04671 L
.71249 .03313 L
.7138 .02721 L
.71499 .0228 L
.71623 .01922 L
.71691 .01769 L
.71755 .01654 L
.71869 .01518 L
.71993 .01472 L
.72059 .01489 L
.7213 .01542 L
.72205 .01635 L
.72275 .01756 L
.724 .02056 L
.72535 .02496 L
.72804 .03734 L
.73097 .05618 L
.73632 .10378 L
Mistroke
.74635 .22981 L
.75733 .39025 L
.76272 .46256 L
.76767 .51892 L
.77219 .55907 L
.77446 .57469 L
.77696 .58802 L
.77827 .5934 L
.77946 .5973 L
.78013 .59904 L
.78084 .6006 L
.78215 .60254 L
.78337 .60329 L
.78467 .60298 L
.78578 .60181 L
.787 .59955 L
.78818 .5964 L
.78947 .5919 L
.79207 .57947 L
.79448 .56407 L
.79672 .54653 L
.80178 .49637 L
.80719 .42934 L
.82614 .16034 L
.83112 .10285 L
.83638 .05607 L
.83914 .03827 L
.8406 .0308 L
.84216 .02433 L
.8429 .02186 L
.8436 .01984 L
.84492 .01696 L
.84615 .01532 L
.84684 .01485 L
.84749 .01472 L
.84876 .01527 L
.84947 .01607 L
.85013 .01711 L
.85134 .01978 L
.85261 .02368 L
.85483 .03301 L
.85725 .0469 L
.86263 .09066 L
.86764 .1457 L
.88632 .40956 L
.89144 .47613 L
.89687 .53429 L
.9015 .57104 L
.90414 .58599 L
.90542 .59162 L
Mistroke
.90661 .59587 L
.90792 .59946 L
.90862 .60088 L
.90935 .60205 L
.91006 .60281 L
.91072 .60321 L
.91196 .60317 L
.91319 .60209 L
.91432 .6002 L
.91555 .59713 L
.91686 .59276 L
.91958 .58006 L
.92212 .56388 L
.9279 .51266 L
.93761 .39055 L
.94803 .23811 L
.95495 .14548 L
.95861 .10418 L
.96249 .06787 L
.96587 .04348 L
.96779 .0328 L
.9696 .02499 L
.97041 .02221 L
.97126 .01975 L
.97199 .01801 L
.97277 .01656 L
.97362 .01546 L
.9744 .01488 L
.97534 .01473 L
.97619 .01512 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[28]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`006`3oool00`000000oooo0?ooo`3o0?ooo`<0oooo
000C0?ooo`D000000`3oool00`000000oooo0?ooo`3o0?ooo`<0oooo000E0?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool0o`3oool30?ooo`005@3oool00`000000oooo0?ooo`030?oo
o`8000003@3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo01h0oooo00<000000?oo
o`3oool07`3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo01h0oooo00<000000?oo
o`3oool07P3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo00/0oooo000=0?ooo`@0
0000103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00/0oooo0P00000O0?ooo`<0
00007P3oool3000001l0oooo0P00000O0?ooo`8000007`3oool3000001h0oooo0`00000N0?ooo`<0
0000303oool001D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0000001h0oooo00<000000?ooo`0000007P3oool00`000000oooo0000000N0?ooo`800000
00<0oooo0000003oool0703oool2000000030?ooo`000000oooo01d0oooo00<000000?ooo`000000
7P3oool00`000000oooo0000000N0?ooo`030000003oool0oooo00`0oooo000D0?ooo`8000001@3o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0000001h0oooo00@000000?ooo`3oool00000
7@3oool010000000oooo0?ooo`00000M0?ooo`040000003oool0oooo000001d0oooo00@000000?oo
o`3oool000007P3oool00`000000oooo0000000N0?ooo`030000003oool0000001h0oooo00<00000
0?ooo`3oool0303oool001/0oooo00<000000?ooo`3oool02P3oool010000000oooo0?ooo`00000M
0?ooo`050000003oool0oooo0?ooo`0000007@3oool010000000oooo0?ooo`00000M0?ooo`040000
003oool0oooo000001d0oooo00@000000?ooo`3oool000007@3oool010000000oooo0?ooo`00000M
0?ooo`040000003oool0oooo000001d0oooo00<000000?ooo`3oool03@3oool001/0oooo00<00000
0?ooo`3oool02P3oool01@000000oooo0?ooo`3oool0000001`0oooo00D000000?ooo`3oool0oooo
0000000M0?ooo`040000003oool0oooo000001d0oooo00@000000?ooo`3oool000007@3oool01000
0000oooo0?ooo`00000M0?ooo`050000003oool0oooo0?ooo`000000703oool01@000000oooo0?oo
o`3oool0000001`0oooo00<000000?ooo`3oool03@3oool001/0oooo00<000000?ooo`3oool02P3o
ool01@000000oooo0?ooo`3oool0000001`0oooo00D000000?ooo`3oool0oooo0000000L0?ooo`05
0000003oool0oooo0?ooo`000000703oool01@000000oooo0?ooo`3oool0000001`0oooo00D00000
0?ooo`3oool0oooo0000000M0?ooo`050000003oool0oooo0?ooo`000000703oool01@000000oooo
0?ooo`3oool0000001`0oooo00<000000?ooo`3oool03@3oool001/0oooo00<000000?ooo`3oool0
2P3oool01@000000oooo0?ooo`3oool0000001`0oooo00D000000?ooo`3oool0oooo0000000L0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0I0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool06P3oool01@000000oooo0?ooo`3oool0
000001`0oooo00<000000?ooo`3oool03@3oool001/0oooo0P00000;0?ooo`050000003oool0oooo
0?ooo`000000703oool01@000000oooo0?ooo`3oool0000001`0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3o
ool00`000000oooo0?ooo`0=0?ooo`006`3oool00`000000oooo0?ooo`0:0?ooo`050000003oool0
oooo0?ooo`000000703oool01@000000oooo0?ooo`3oool0000001`0oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
6@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
6@3oool00`000000oooo0?ooo`0=0?ooo`006`3oool00`000000oooo0?ooo`090?ooo`030000003o
ool0oooo0080oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo01P0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo01T0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0=0?ooo`006`3oool0
0`000000oooo0?ooo`090?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool06@3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0603oool0
0`000000oooo0?ooo`0>0?ooo`006`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
01P0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
01P0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0>0?ooo`006`3oool00`000000
oooo0?ooo`090?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0603oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`0H0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0603oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0603oool00`000000
oooo0?ooo`0>0?ooo`006`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool0603oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00@0oooo
00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01L0oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0>0?ooo`006`3oool00`000000oooo0?oo
o`080?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0603oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?oo
o`0H0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?oo
o`0H0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?oo
o`0>0?ooo`006`3oool2000000T0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0G
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`03
0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0G
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0G
0?ooo`030000003oool0oooo00h0oooo000K0?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool05`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool05`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01L0oooo00<000000?oo
o`3oool01@3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00h0oooo000K0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0G0?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo01L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0G0?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0G0?ooo`03
0000003oool0oooo00h0oooo000K0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
5`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0
1@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
5`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0
1@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00l0oooo000K0?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0G0?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`050?ooo`030000003o
ool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003o
ool0oooo00l0oooo000K0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05`3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05`3oool0
0`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00l0oooo000K0?ooo`030000003oool0oooo
00P0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo
00D0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo
00D0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo
00l0oooo000K0?ooo`8000002@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0
oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0
oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0
oooo00<000000?ooo`3oool03`3oool001/0oooo00<000000?ooo`3oool0203oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05P3oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05P3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool03`3oool001/0oooo00<0
00000?ooo`3oool01`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01H0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<0
00000?ooo`3oool05P3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<0
00000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<0
00000?ooo`3oool03`3oool001/0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0E
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool03`3oool001/0oooo00<000000?oo
o`3oool01`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool01P3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool05P3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool01P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01H0oooo00<000000?oo
o`3oool03`3oool001/0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0F0?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0E0?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo01D0oooo00<000000?ooo`3oool03`3oool001/0oooo00<000000?ooo`3oool0
1`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0
1P3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0
5@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0
1P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0
5@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0
403oool001/0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo01D0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003o
ool0oooo00L0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0E0?ooo`030000003o
ool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`030000003o
ool0oooo01@0oooo00<000000?ooo`3oool0403oool001/0oooo0P0000080?ooo`030000003oool0
oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`030000003oool0
oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0
oooo00H0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`030000003oool0
oooo01D0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0
oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`0@0?ooo`006`3oool00`00
0000oooo0?ooo`070?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`00
0000oooo0?ooo`070?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0503oool00`00
0000oooo0?ooo`080?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`0D0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`00
0000oooo0?ooo`0@0?ooo`006`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01@0
oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01@0
oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`0@0?ooo`006`3oool00`000000oooo
0?ooo`070?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo
0?ooo`070?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo
0?ooo`0D0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo
0?ooo`080?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo
0?ooo`0D0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo
0?ooo`0@0?ooo`006`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00L0oooo00<0
00000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01@0oooo00<0
00000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01@0oooo00<0
00000?ooo`3oool01`3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0503oool00`000000oooo0?ooo`0@0?ooo`006`3oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C
0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0C
0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`0@
0?ooo`00203oool2000000@0oooo0P0000040?ooo`8000001@3oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo01@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0C0?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`0@0?oo
o`001`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000000@0oooo00<00000
0?ooo`3oool01P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01@0oooo00<00000
0?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01@0oooo00<00000
0?ooo`3oool0403oool000L0oooo00@000000?ooo`3oool000002`3oool00`000000oooo0?ooo`02
0?ooo`8000001`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01<0oooo00<00000
0?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01@0oooo00<00000
0?ooo`3oool0403oool000030?ooo`0000000000008000000P3oool010000000oooo0?ooo`000009
0?ooo`8000001@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01<0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool0503oool00`000000oooo0?ooo`0@0?ooo`001`3oool010000000oooo0?ooo`000009
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`09
0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0C
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool04@3oool000P0oooo0P00000:0?oo
o`<00000103oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool04P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool04`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0180oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool04P3oool00`000000oooo0?ooo`0A0?ooo`006`3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo01<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0C0?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0A0?ooo`00
6`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
4P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
4P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
4P3oool00`000000oooo0?ooo`0A0?ooo`006`3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo00T0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003o
ool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0A0?ooo`006`3oool2
000000L0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0
oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo01<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0
oooo00X0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0
oooo0140oooo000K0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`00
0000oooo0?ooo`090?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool04P3oool00`00
0000oooo0?ooo`090?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0B0?ooo`030000003oool0oooo0140oooo000K0?ooo`030000003oool0oooo00H0
oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0
oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01<0
oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0140
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0140
oooo000K0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool02P3oool00`000000oooo
0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo
0?ooo`090?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool02P3oool00`000000oooo
0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool04P3oool00`000000oooo
0?ooo`090?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo
0?ooo`0B0?ooo`030000003oool0oooo0140oooo000K0?ooo`030000003oool0oooo00D0oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool04P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0140oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<0
00000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0140oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0140oooo000K
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0A
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A
0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0B
0?ooo`030000003oool0oooo0140oooo000K0?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool02`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0140oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0140oooo000K0?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0A0?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`03
0000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`03
0000003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`03
0000003oool0oooo0100oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`03
0000003oool0oooo0180oooo000K0?ooo`8000001P3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo0140oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo0140oooo00<000000?ooo`3oool04P3oool001/0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02@3o
ool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0403o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool04P3o
ool001/0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo0140oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0
oooo00/0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0
oooo00/0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool04P3oool001/0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0A0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`00
0000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0403oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool04P3oool001/0
oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool04P3oool001/0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo
0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0303oool00`000000oooo
0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0403oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool04P3oool001/0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0100oooo00<0
00000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<0
00000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00l0oooo00<0
00000?ooo`3oool0303oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<0
00000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<0
00000?ooo`3oool04P3oool001/0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0<
0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool04P3oool001/0oooo0P0000050?oo
o`030000003oool0oooo00`0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00/0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?oo
o`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?oo
o`030000003oool0oooo00/0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0B0?oo
o`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3o
ool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3o
ool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3o
ool0303oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool0403oool00`000000oooo0?ooo`0B0?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo00`0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo00l0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0@0?ooo`030000
003oool0oooo00`0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000
003oool0oooo00l0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0B0?ooo`006`3o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool03`3o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0303o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03`3o
ool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0403o
ool00`000000oooo0?ooo`0B0?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool0303oool5000000040?ooo`0000000000000000<0oooo0`00
00020?ooo`040000003oool00000000000h0oooo00<000000?ooo`3oool0303oool5000000040?oo
o`0000000000000000D0oooo00<000000?ooo`3oool00P3oool2000000d0oooo00<000000?ooo`3o
ool0303oool5000000040?ooo`0000000000000000@0oooo0P0000030?ooo`<000003`3oool00`00
0000oooo0?ooo`0;0?ooo`D0000000@0oooo00000000000000000`3oool3000000@0oooo0P00000=
0?ooo`030000003oool0oooo00d0oooo100000030?ooo`800000103oool2000000@0oooo0P000003
0?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00d0oooo00<000000?oo
o`3oool03P3oool2000000@0oooo00D000000?ooo`3oool0oooo000000020?ooo`040000003oool0
000000000080oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0<0?ooo`030000003o
ool0000000D0oooo00<000000?ooo`3oool00`3oool010000000oooo0?ooo`3oool2000000030?oo
o`000000oooo00/0oooo00<000000?ooo`3oool03P3oool2000000@0oooo00D000000?ooo`3oool0
oooo000000020?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0303oool00`00
0000oooo0?ooo`0=0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00`3oool01000
0000oooo0?ooo`3oool2000000030?ooo`000000oooo00/0oooo00<000000?ooo`3oool03@3oool0
0`000000oooo000000030?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000
0P3oool010000000oooo0?ooo`0000020?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool2000000<0oooo0`0000020?ooo`050000
003oool0oooo0000003oool00P0000020?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool03@3oool2000000@0oooo0`0000040?ooo`040000003oool0oooo0?ooo`80000000<0oooo0000
003oool02`3oool00`000000oooo0?ooo`0>0?ooo`8000000`3oool300000080oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`00000>0?ooo`030000003oool0oooo00d0oooo00<0
00000?ooo`3oool00P3oool3000000<0oooo0`0000020?ooo`80000000<0oooo0000003oool02`3o
ool00`000000oooo0?ooo`0>0?ooo`8000000`3oool010000000oooo0?ooo`0000020?ooo`040000
003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool001/0oooo0P0000050?ooo`03
0000003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool01@000000oooo0?ooo`3oool00000
0080oooo00<000000?ooo`3oool00`0000020?ooo`8000000P3oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo00d0oooo0P0000030?ooo`040000003oool0oooo000000D0oooo00<00000
0?ooo`3oool00P0000000`3oool000000?ooo`0;0?ooo`030000003oool0oooo00h0oooo0P000002
0?ooo`040000003oool0oooo000000<0oooo0P0000030?ooo`040000003oool0oooo000000h0oooo
00<000000?ooo`3oool03@3oool01@000000oooo0?ooo`3oool000000080oooo00@000000?ooo`3o
ool000000P3oool00`000000oooo0?ooo`02000000030?ooo`000000oooo00/0oooo00<000000?oo
o`3oool03`3oool01@000000oooo0?ooo`3oool000000080oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`4000000@3oool10?ooo`006`3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool01@000000oooo0?oo
o`3oool000000080oooo00D000000?ooo`3oool0oooo000000030?ooo`8000000P3oool00`000000
oooo0?ooo`0;0?ooo`030000003oool0oooo00d0oooo0P0000030?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool000000P3oool2000000030?ooo`000000oooo00/0oooo00<00000
0?ooo`3oool03P3oool200000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo000000h0oooo00<000000?ooo`3oool03@3oool01@000000
oooo0?ooo`3oool000000080oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`02
000000030?ooo`000000oooo00`0oooo00<000000?ooo`3oool0303oool010000000oooo00000000
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000
oooo0?ooo`0000020?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool03@3oool2000000@0oooo0`0000030?ooo`<0000000<0oooo0000003o
ool00P00000>0?ooo`030000003oool0oooo00d0oooo0P0000040?ooo`<000000P3oool4000000<0
oooo0P00000=0?ooo`030000003oool0oooo00d0oooo0`0000030?ooo`<000000`3oool2000000<0
oooo0`00000?0?ooo`030000003oool0oooo00`0oooo0P0000040?ooo`<000000`3oool3000000<0
oooo0P00000>0?ooo`030000003oool0oooo00d0oooo0P0000040?ooo`800000103oool2000000@0
oooo0P0000030?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00d0oooo
00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00h0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00d0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00l0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00d0oooo
00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0C0?ooo`006`3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?oo
o`0=0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?oo
o`0?0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?oo
o`0>0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?oo
o`0>0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?oo
o`0C0?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00d0oooo00<00000
0?ooo`3oool03P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00h0oooo00<00000
0?ooo`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00d0oooo00<00000
0?ooo`3oool03@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00h0oooo00<00000
0?ooo`3oool03@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00d0oooo00<00000
0?ooo`3oool03P3oool00`000000oooo0?ooo`0C0?ooo`006`3oool00`000000oooo0?ooo`030?oo
o`030000003oool0oooo00h0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0=0?oo
o`030000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0?0?oo
o`030000003oool0oooo00d0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0>0?oo
o`030000003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0>0?oo
o`030000003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0C0?oo
o`006`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3o
ool03P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3o
ool03@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3o
ool03@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3o
ool03P3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3o
ool03P3oool00`000000oooo0?ooo`0C0?ooo`005@3ooooo000000H000001P3oool001/0oooo00<0
00000?ooo`3oool00`3oool01@000000oooo0?ooo`3oool0000000T0oooo00@000000?ooo`3oool0
00001@3oool00`000000oooo0?ooo`070?ooo`800000203oool00`000000oooo0?ooo`040?ooo`04
0000003oool0oooo000000T0oooo00D000000?ooo`3oool0oooo000000050?ooo`030000003oool0
oooo00L0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`040?ooo`040000003oool0
oooo000000T0oooo00D000000?ooo`3oool0oooo000000050?ooo`030000003oool0oooo00H0oooo
0P0000080?ooo`030000003oool0oooo00D0oooo00<000000?ooo`0000002@3oool01@000000oooo
0?ooo`3oool0000000D0oooo00<000000?ooo`3oool01`3oool00`000000oooo000000070?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool00000
00D0oooo00<000000?ooo`3oool01`3oool2000000P0oooo00<000000?ooo`3oool02P3oool001/0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00H0
oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00l0
oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00X0
oooo000K0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool03`3oool00`000000oooo
0?ooo`0=0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool03@3oool00`000000oooo
0?ooo`0>0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool03P3oool00`000000oooo
0?ooo`0=0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo
0?ooo`0?0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool03@3oool00`000000oooo
0?ooo`0=0?ooo`030000003oool0oooo01@0oooo000K0?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00h0oooo00<0
00000?ooo`3oool03@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00h0oooo00<0
00000?ooo`3oool03P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00h0oooo00<0
00000?ooo`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00d0oooo00<0
00000?ooo`3oool03@3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo01@0oooo000K
0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0>
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0=
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0?
0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0=
0?ooo`030000003oool0oooo01@0oooo000K0?ooo`030000003oool0oooo00<0oooo00<000000?oo
o`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00h0oooo00<000000?oo
o`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00d0oooo00<000000?oo
o`3oool03P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00h0oooo00<000000?oo
o`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<000000?oo
o`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo01@0oooo000K0?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`03
0000003oool0oooo00d0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo00h0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0?0?ooo`03
0000003oool0oooo00`0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo01@0oooo000K0?ooo`800000103oool00`000000oooo0?ooo`0?0?ooo`030000
003oool0oooo00`0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0=0?ooo`030000
003oool0oooo00l0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0>0?ooo`030000
003oool0oooo00`0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0=0?ooo`030000
003oool0oooo00l0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0?0?ooo`030000
003oool0oooo00`0oooo00<000000?ooo`3oool0503oool001/0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03`3o
ool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03`3o
ool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0303o
ool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0503o
ool001/0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo00l0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo00l0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0?0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0503oool001/0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`0<0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool03@3oool00`00
0000oooo0?ooo`0>0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`0=0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0303oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0503oool001/0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0
oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00l0
oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0
oooo00<000000?ooo`3oool0503oool001/0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03`3oool00`000000oooo
0?ooo`0<0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0303oool00`000000oooo
0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0403oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0303oool00`000000oooo
0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0503oool001/0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<0
00000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00l0oooo00<0
00000?ooo`3oool0303oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<0
00000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<0
00000?ooo`3oool0303oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<0
00000?ooo`3oool0503oool001/0oooo0P0000040?ooo`030000003oool0oooo00l0oooo00<00000
0?ooo`3oool0303oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00`0oooo00<00000
0?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0100oooo00<00000
0?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<00000
0?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00l0oooo00<00000
0?ooo`3oool02`3oool00`000000oooo0?ooo`0E0?ooo`006`3oool00`000000oooo0?ooo`030?oo
o`030000003oool0oooo00l0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0?0?oo
o`030000003oool0oooo00`0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?oo
o`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?oo
o`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo00l0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0E0?oo
o`006`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool02P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3o
ool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`0E0?ooo`006`3oool00`000000oooo0?ooo`020?ooo`030000
003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0@0?ooo`030000
003oool0oooo00`0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo00l0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0E0?ooo`006`3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0403o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool02`3o
ool00`000000oooo0?ooo`0E0?ooo`006`3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0
oooo00/0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0E0?ooo`006`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02`3oool00`00
0000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04@3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0E0?ooo`006`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0140
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0E0?ooo`006`3oool2000000<0oooo
00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0140oooo
00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00/0oooo
00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo
00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo
00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo01D0oooo
000K0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool04@3oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?oo
o`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool02P3oool00`000000oooo0?oo
o`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo01D0oooo000K0?ooo`030000003oool0oooo0080oooo00<00000
0?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0140oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0180oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo01D0oooo000K0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo0180oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo01D0oooo000K0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3o
ool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3o
ool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3o
ool02P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3o
ool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo01D0oooo000K0?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000
003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo0180oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0A0?ooo`030000
003oool0oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo01D0oooo000K0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool04@3o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool04P3o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool04@3o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01H0oooo000K0?ooo`030000003oool0
oooo0080oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo0180oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0
oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0
oooo00X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo01H0oooo000K0?ooo`8000000P3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo
00P0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo
0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo
00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo
01<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo
00T0oooo00<000000?ooo`3oool05P3oool001/0oooo00D000000?ooo`3oool0oooo0000000E0?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0B0?oo
o`030000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?oo
o`030000003oool0oooo01<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00T0oooo00<000000?ooo`3oool05P3oool001/0oooo00D000000?ooo`3o
ool0oooo0000000E0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool04P3oool00`00
0000oooo0?ooo`0:0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool04P3oool00`00
0000oooo0?ooo`090?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`0A0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool05P3oool001/0
oooo00D000000?ooo`3oool0oooo0000000E0?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool04`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0180oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01<0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool05P3oool001/0oooo00D000000?ooo`3oool0oooo0000000E0?ooo`030000003oool0oooo
00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo
01<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo
00T0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo
01<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo
00T0oooo00<000000?ooo`3oool05P3oool000P0oooo0P0000040?ooo`800000103oool2000000D0
oooo00D000000?ooo`3oool0oooo0000000E0?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<000000?oo
o`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<000000?oo
o`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool05P3oool000L0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`000004
0?ooo`050000003oool0oooo0?ooo`0000005@3oool00`000000oooo0?ooo`080?ooo`030000003o
ool0oooo01<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`090?ooo`030000003o
ool0oooo01<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003o
ool0oooo01H0oooo00070?ooo`040000003oool0oooo000000/0oooo00<000000?ooo`3oool00P3o
ool200000080oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003o
ool0oooo01<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`090?ooo`030000003o
ool0oooo01<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003o
ool0oooo01H0oooo00070?ooo`040000003oool0oooo000000T0oooo0P0000050?ooo`040000003o
ool0oooo000001H0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000
003oool0oooo01<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0C0?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000
003oool0oooo01<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0F0?ooo`001`3o
ool010000000oooo0?ooo`0000090?ooo`030000003oool0oooo00@0oooo00@000000?ooo`3oool0
00005P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3o
ool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3o
ool0203oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01H0oooo00080?ooo`800000
2P3oool3000000@0oooo00@000000?ooo`3oool000005P3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo01@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo01<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo01H0oooo000K0?ooo`040000003oool0oooo000001H0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3o
ool0203oool00`000000oooo0?ooo`0F0?ooo`006`3oool010000000oooo0?ooo`00000F0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo01@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0C0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool05P3oool001/0oooo00@000000?ooo`3oool0
00005P3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3o
ool0503oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01H0oooo000K0?ooo`040000
003oool0oooo000001H0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0G0?ooo`00
6`3oool2000000030?ooo`000000oooo01D0oooo00<000000?ooo`3oool01`3oool00`000000oooo
0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool01`3oool00`000000oooo
0?ooo`0D0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo
0?ooo`070?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo
0?ooo`0G0?ooo`006`3oool010000000oooo0?ooo`00000G0?ooo`030000003oool0oooo00H0oooo
00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo
00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00L0oooo
00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01@0oooo
00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00L0oooo
00<000000?ooo`3oool05`3oool001/0oooo00@000000?ooo`3oool000005`3oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo
0?ooo`070?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo
0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo01L0oooo000K0?ooo`040000003oool0oooo000001L0oooo
00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo
00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo
00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00L0oooo
00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo
00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0G0?ooo`006`3oool010000000oooo0?oo
o`00000G0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`0F0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05`3oool001/0oooo00<0
00000?ooo`000000603oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<0
00000?ooo`3oool05P3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00H0oooo00<0
00000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01L0oooo000K
0?ooo`030000003oool0000001P0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0E
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0E
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0G
0?ooo`006`3oool00`000000oooo0000000H0?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool01P3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool05@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?oo
o`3oool01P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool05`3oool001/0oooo0`00000H0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3o
ool01P3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3o
ool01P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3o
ool05`3oool001/0oooo00<000000?ooo`000000603oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`030000
003oool0oooo01L0oooo000K0?ooo`030000003oool0000001P0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05`3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05P3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`0G0?ooo`006`3oool00`000000oooo0000000H0?ooo`030000003oool0
oooo00D0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0
oooo01L0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0
oooo00D0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0
oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0
oooo00@0oooo00<000000?ooo`3oool0603oool001/0oooo00<000000?ooo`0000006@3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05`3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo01P0oooo000K0?ooo`030000003oool0000001T0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01L0
oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01L0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0H0?ooo`006`3oool00`000000oooo
0000000I0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo
0?ooo`040?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0603oool001/0oooo0`00
000I0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?oo
o`0G0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0103oool00`000000oooo0?oo
o`0G0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0603oool001/0oooo00<00000
0?ooo`0000006@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01L0oooo00<00000
0?ooo`3oool0103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<00000
0?ooo`3oool05`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01P0oooo00<00000
0?ooo`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00@0oooo00<00000
0?ooo`3oool05`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01P0oooo000K0?oo
o`8000006P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01L0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool05`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01P0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool05`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01T0oooo000K0?ooo`80
00006P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3o
ool0603oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool06@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo000K0?ooo`800000
6`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
603oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
6@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo000K0?ooo`8000006`3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0603o
ool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool06@3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo000K0?ooo`8000006`3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool06@3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo000K0?ooo`8000006`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo000K0?ooo`8000006`3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool00`000000
oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06P3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo01T0oooo00D000000?ooo`3oool0oooo0000000L0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo01T0oooo000K0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool06@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01X0oooo00D000000?ooo`3o
ool0oooo0000000L0?ooo`050000003oool0oooo0?ooo`000000703oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0I
0?ooo`006`3oool00`000000oooo0?ooo`0J0?ooo`050000003oool0oooo0?ooo`0000007@3oool0
10000000oooo0?ooo`00000L0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06P3o
ool01@000000oooo0?ooo`3oool0000001`0oooo00D000000?ooo`3oool0oooo0000000L0?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool06P3oool01@000000oooo0?ooo`3oool00000
01/0oooo000K0?ooo`030000003oool0oooo01X0oooo00D000000?ooo`3oool0oooo0000000M0?oo
o`040000003oool0oooo000001d0oooo00@000000?ooo`3oool000007@3oool01@000000oooo0?oo
o`3oool0000001`0oooo00D000000?ooo`3oool0oooo0000000M0?ooo`040000003oool0oooo0000
01d0oooo00D000000?ooo`3oool0oooo0000000K0?ooo`006`3oool00`000000oooo0?ooo`0K0?oo
o`040000003oool0oooo000001d0oooo00@000000?ooo`3oool000007@3oool010000000oooo0?oo
o`00000M0?ooo`040000003oool0oooo000001h0oooo00@000000?ooo`3oool000007@3oool01000
0000oooo0?ooo`00000M0?ooo`040000003oool0oooo000001`0oooo000K0?ooo`030000003oool0
oooo01/0oooo00@000000?ooo`3oool000007@3oool010000000oooo0?ooo`00000M0?ooo`040000
003oool0oooo000001d0oooo00@000000?ooo`3oool000007P3oool010000000oooo0?ooo`00000M
0?ooo`040000003oool0oooo000001d0oooo00@000000?ooo`3oool00000703oool001<0oooo1@00
00030?ooo`030000003oool0oooo01/0oooo00@000000?ooo`3oool000007P3oool00`000000oooo
0000000N0?ooo`030000003oool0000001h0oooo00<000000?ooo`0000007P3oool00`000000oooo
0000000N0?ooo`040000003oool0oooo000001d0oooo00@000000?ooo`3oool00000703oool001D0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0L0?ooo`8000007`3oool2000001l0
oooo0P00000O0?ooo`<000007P3oool3000001l0oooo0P00000O0?ooo`8000007@3oool001D0oooo
00<000000?ooo`3oool00`3oool2000001d0oooo0P00000P0?ooo`030000003oool0oooo01h0oooo
00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo01h0oooo
00<000000?ooo`3oool07P3oool2000001d0oooo000E0?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool0o`3oool30?ooo`005@3oool00`000000oooo0?ooo`030?ooo`030000003oool0
oooo0?l0oooo0`3oool001@0oooo0P0000050?ooo`030000003oool0oooo0?l0oooo0`3oool00?l0
oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0
oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {1944.5, -1.15557, 0.20182, \
0.0130619}}]
}, Open  ]],

Cell[TextData[{
  "This shows a plot of the invariant at the ends of the time steps ",
  StyleBox["NDSolve", "MR"],
  " took."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(ts\  = \ 
      First[InterpolatingFunctionCoordinates[
          x\  /. \ dsol]]; \), "\[IndentingNewLine]", 
    \(ListPlot[
      Transpose[{ts, \ \(invariant\  + 2\ Cos[1] /. dsol\)\  /. \ 
            t \[Rule] ts}], \ PlotRange \[Rule] All]; \)}], "Input",
  CellLabel->"In[29]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.00047619 0.603292 1.8434e+006 [
[.2619 .59079 -9 -9 ]
[.2619 .59079 9 0 ]
[.5 .59079 -12 -9 ]
[.5 .59079 12 0 ]
[.7381 .59079 -12 -9 ]
[.7381 .59079 12 0 ]
[.97619 .59079 -12 -9 ]
[.97619 .59079 12 0 ]
[.01131 .05027 -39 -6.25 ]
[.01131 .05027 0 6.25 ]
[.01131 .14244 -51 -6.25 ]
[.01131 .14244 0 6.25 ]
[.01131 .23461 -39 -6.25 ]
[.01131 .23461 0 6.25 ]
[.01131 .32678 -51 -6.25 ]
[.01131 .32678 0 6.25 ]
[.01131 .41895 -39 -6.25 ]
[.01131 .41895 0 6.25 ]
[.01131 .51112 -39 -6.25 ]
[.01131 .51112 0 6.25 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.2619 .60329 m
.2619 .60954 L
s
[(500)] .2619 .59079 0 1 Mshowa
.5 .60329 m
.5 .60954 L
s
[(1000)] .5 .59079 0 1 Mshowa
.7381 .60329 m
.7381 .60954 L
s
[(1500)] .7381 .59079 0 1 Mshowa
.97619 .60329 m
.97619 .60954 L
s
[(2000)] .97619 .59079 0 1 Mshowa
.125 Mabswid
.07143 .60329 m
.07143 .60704 L
s
.11905 .60329 m
.11905 .60704 L
s
.16667 .60329 m
.16667 .60704 L
s
.21429 .60329 m
.21429 .60704 L
s
.30952 .60329 m
.30952 .60704 L
s
.35714 .60329 m
.35714 .60704 L
s
.40476 .60329 m
.40476 .60704 L
s
.45238 .60329 m
.45238 .60704 L
s
.54762 .60329 m
.54762 .60704 L
s
.59524 .60329 m
.59524 .60704 L
s
.64286 .60329 m
.64286 .60704 L
s
.69048 .60329 m
.69048 .60704 L
s
.78571 .60329 m
.78571 .60704 L
s
.83333 .60329 m
.83333 .60704 L
s
.88095 .60329 m
.88095 .60704 L
s
.92857 .60329 m
.92857 .60704 L
s
.25 Mabswid
0 .60329 m
1 .60329 L
s
.02381 .05027 m
.03006 .05027 L
s
gsave
.01131 .05027 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-3) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .14244 m
.03006 .14244 L
s
gsave
.01131 .14244 -51 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-2.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .23461 m
.03006 .23461 L
s
gsave
.01131 .23461 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .32678 m
.03006 .32678 L
s
gsave
.01131 .32678 -51 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-1.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .41895 m
.03006 .41895 L
s
gsave
.01131 .41895 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .51112 m
.03006 .51112 L
s
gsave
.01131 .51112 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-8) show
grestore
.125 Mabswid
.02381 .06871 m
.02756 .06871 L
s
.02381 .08714 m
.02756 .08714 L
s
.02381 .10558 m
.02756 .10558 L
s
.02381 .12401 m
.02756 .12401 L
s
.02381 .16088 m
.02756 .16088 L
s
.02381 .17931 m
.02756 .17931 L
s
.02381 .19774 m
.02756 .19774 L
s
.02381 .21618 m
.02756 .21618 L
s
.02381 .25305 m
.02756 .25305 L
s
.02381 .27148 m
.02756 .27148 L
s
.02381 .28991 m
.02756 .28991 L
s
.02381 .30835 m
.02756 .30835 L
s
.02381 .34522 m
.02756 .34522 L
s
.02381 .36365 m
.02756 .36365 L
s
.02381 .38208 m
.02756 .38208 L
s
.02381 .40052 m
.02756 .40052 L
s
.02381 .43739 m
.02756 .43739 L
s
.02381 .45582 m
.02756 .45582 L
s
.02381 .47425 m
.02756 .47425 L
s
.02381 .49269 m
.02756 .49269 L
s
.02381 .52956 m
.02756 .52956 L
s
.02381 .54799 m
.02756 .54799 L
s
.02381 .56642 m
.02756 .56642 L
s
.02381 .58486 m
.02756 .58486 L
s
.02381 .03184 m
.02756 .03184 L
s
.02381 .01341 m
.02756 .01341 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.008 w
.02381 .60329 Mdot
.02389 .60329 Mdot
.02402 .6033 Mdot
.02414 .6033 Mdot
.02428 .60329 Mdot
.02441 .60325 Mdot
.02454 .6032 Mdot
.02469 .60318 Mdot
.02482 .60309 Mdot
.02497 .60297 Mdot
.02512 .60295 Mdot
.02529 .60304 Mdot
.02546 .60305 Mdot
.02564 .60318 Mdot
.02582 .60332 Mdot
.02599 .60283 Mdot
.02616 .60259 Mdot
.02632 .60242 Mdot
.02648 .6021 Mdot
.02664 .60189 Mdot
.02682 .60202 Mdot
.02699 .60209 Mdot
.02717 .60214 Mdot
.02737 .60271 Mdot
.02755 .6022 Mdot
.02772 .60177 Mdot
.02789 .60165 Mdot
.02805 .60127 Mdot
.02821 .60098 Mdot
.02839 .6011 Mdot
.02857 .6012 Mdot
.02875 .60123 Mdot
.02895 .60183 Mdot
.02913 .60141 Mdot
.0293 .60089 Mdot
.02947 .60079 Mdot
.02963 .60041 Mdot
.02979 .6001 Mdot
.02996 .60016 Mdot
.03014 .6003 Mdot
.03032 .60032 Mdot
.03052 .6009 Mdot
.0307 .60059 Mdot
.03088 .6 Mdot
.03104 .59991 Mdot
.0312 .59953 Mdot
.03137 .5992 Mdot
.03154 .59921 Mdot
.03172 .59937 Mdot
.03189 .59938 Mdot
.03209 .5999 Mdot
.03227 .59972 Mdot
.03245 .59906 Mdot
.03261 .59895 Mdot
.03278 .59859 Mdot
.03294 .59824 Mdot
.03311 .5982 Mdot
.03328 .59837 Mdot
.03346 .59838 Mdot
.03366 .59875 Mdot
.03384 .59871 Mdot
.03402 .59803 Mdot
.03418 .59788 Mdot
.03435 .59755 Mdot
.03451 .59719 Mdot
.03468 .59708 Mdot
.03485 .59726 Mdot
.03503 .59728 Mdot
.03522 .59749 Mdot
.03541 .59763 Mdot
.03558 .59698 Mdot
.03575 .59675 Mdot
.03592 .59647 Mdot
.03608 .59611 Mdot
.03625 .59593 Mdot
.03642 .59611 Mdot
.0366 .59614 Mdot
.03678 .59627 Mdot
.03698 .59661 Mdot
.03716 .596 Mdot
.03733 .59568 Mdot
.0375 .59548 Mdot
.03765 .5951 Mdot
.03782 .59488 Mdot
.03799 .59504 Mdot
.03817 .59509 Mdot
.03836 .59517 Mdot
.03856 .59575 Mdot
.03874 .59518 Mdot
.03891 .59478 Mdot
.03908 .59462 Mdot
.03923 .59423 Mdot
.0394 .59397 Mdot
.03958 .59412 Mdot
.03975 .5942 Mdot
.03994 .59425 Mdot
.04014 .59485 Mdot
.04032 .59432 Mdot
.04049 .59387 Mdot
.04066 .59374 Mdot
.04082 .59336 Mdot
.04098 .59307 Mdot
.04115 .59319 Mdot
.04133 .59329 Mdot
.04151 .59333 Mdot
.04171 .59393 Mdot
.04189 .59348 Mdot
.04207 .59298 Mdot
.04223 .59288 Mdot
.04239 .5925 Mdot
.04256 .59218 Mdot
.04273 .59226 Mdot
.04291 .59239 Mdot
.04309 .59242 Mdot
.04329 .59301 Mdot
.04347 .59267 Mdot
.04364 .5921 Mdot
.04381 .59201 Mdot
.04397 .59163 Mdot
.04413 .59129 Mdot
.0443 .59132 Mdot
.04448 .59147 Mdot
.04466 .59149 Mdot
.04486 .59202 Mdot
.04504 .59181 Mdot
.04522 .59117 Mdot
.04538 .59107 Mdot
.04555 .5907 Mdot
.04571 .59035 Mdot
.04587 .59033 Mdot
.04605 .59049 Mdot
.04623 .59051 Mdot
.04643 .59092 Mdot
.04661 .59084 Mdot
.04679 .59017 Mdot
.04695 .59003 Mdot
.04712 .58969 Mdot
.04728 .58933 Mdot
.04744 .58924 Mdot
.04762 .58941 Mdot
.0478 .58943 Mdot
.04799 .58968 Mdot
.04818 .58976 Mdot
.04835 .58911 Mdot
.04852 .5889 Mdot
.04869 .58861 Mdot
.04885 .58824 Mdot
.04901 .58809 Mdot
.04919 .58826 Mdot
.04937 .5883 Mdot
.04955 .58844 Mdot
.04975 .58872 Mdot
.04992 .5881 Mdot
.05009 .58781 Mdot
.05026 .58759 Mdot
.05042 .58721 Mdot
.05059 .587 Mdot
.05076 .58716 Mdot
.05094 .58722 Mdot
.05112 .5873 Mdot
.05132 .58783 Mdot
.0515 .58726 Mdot
.05167 .58688 Mdot
.05184 .58671 Mdot
.052 .58632 Mdot
.05216 .58607 Mdot
.05234 .58622 Mdot
.05252 .58629 Mdot
.0527 .58635 Mdot
.0529 .58694 Mdot
.05308 .58639 Mdot
.05325 .58596 Mdot
.05342 .58583 Mdot
.05358 .58544 Mdot
.05374 .58516 Mdot
.05392 .58529 Mdot
.0541 .58538 Mdot
.05428 .58542 Mdot
.05448 .58603 Mdot
.05466 .58555 Mdot
.05483 .58506 Mdot
.055 .58496 Mdot
.05516 .58458 Mdot
.05532 .58427 Mdot
.0555 .58436 Mdot
.05568 .58449 Mdot
.05586 .58451 Mdot
.05606 .58512 Mdot
.05624 .58475 Mdot
.05641 .58419 Mdot
.05658 .5841 Mdot
.05674 .58372 Mdot
.0569 .58339 Mdot
.05707 .58343 Mdot
.05725 .58358 Mdot
.05743 .5836 Mdot
.05763 .58415 Mdot
.05781 .5839 Mdot
.05798 .58328 Mdot
.05815 .58318 Mdot
.05831 .58281 Mdot
.05847 .58246 Mdot
.05864 .58245 Mdot
.05882 .58261 Mdot
.059 .58263 Mdot
.0592 .58311 Mdot
.05938 .58299 Mdot
.05955 .58231 Mdot
.05972 .58218 Mdot
.05989 .58183 Mdot
.06005 .58147 Mdot
.06021 .58141 Mdot
.06039 .58158 Mdot
.06057 .5816 Mdot
.06076 .58188 Mdot
.06095 .58192 Mdot
.06112 .58125 Mdot
.06129 .58106 Mdot
.06146 .58076 Mdot
.06162 .5804 Mdot
.06178 .58026 Mdot
.06196 .58043 Mdot
.06214 .58046 Mdot
.06232 .58063 Mdot
.06252 .58085 Mdot
.06269 .58022 Mdot
.06286 .57995 Mdot
.06303 .57971 Mdot
.06319 .57934 Mdot
.06335 .57914 Mdot
.06353 .57931 Mdot
.06371 .57935 Mdot
.06389 .57945 Mdot
.06409 .5799 Mdot
.06427 .57932 Mdot
.06444 .57896 Mdot
.06461 .57878 Mdot
.06477 .57839 Mdot
.06493 .57815 Mdot
.06511 .57831 Mdot
.06528 .57837 Mdot
.06547 .57844 Mdot
.06567 .57903 Mdot
.06585 .57846 Mdot
.06602 .57804 Mdot
.06619 .5779 Mdot
.06635 .57751 Mdot
.06651 .57724 Mdot
.06669 .57738 Mdot
.06686 .57746 Mdot
.06705 .57751 Mdot
.06725 .57811 Mdot
.06743 .57761 Mdot
.0676 .57714 Mdot
.06777 .57703 Mdot
.06793 .57665 Mdot
.06809 .57635 Mdot
.06826 .57645 Mdot
.06844 .57656 Mdot
.06862 .57659 Mdot
.06882 .5772 Mdot
.069 .5768 Mdot
.06918 .57626 Mdot
.06934 .57617 Mdot
.0695 .57579 Mdot
.06967 .57546 Mdot
.06984 .57552 Mdot
.07002 .57566 Mdot
.0702 .57568 Mdot
.0704 .57624 Mdot
.07058 .57597 Mdot
.07075 .57536 Mdot
.07092 .57526 Mdot
.07108 .57489 Mdot
.07124 .57455 Mdot
.07141 .57455 Mdot
.07159 .57471 Mdot
.07177 .57472 Mdot
.07197 .57522 Mdot
.07215 .57508 Mdot
.07232 .57441 Mdot
.07249 .57429 Mdot
.07265 .57393 Mdot
.07281 .57358 Mdot
.07298 .57352 Mdot
.07316 .57369 Mdot
.07334 .57371 Mdot
.07353 .57402 Mdot
.07372 .57403 Mdot
.07389 .57336 Mdot
.07406 .57318 Mdot
.07422 .57287 Mdot
.07438 .57251 Mdot
.07455 .57238 Mdot
.07473 .57256 Mdot
.0749 .57259 Mdot
.07509 .57277 Mdot
.07528 .57295 Mdot
.07546 .57232 Mdot
.07563 .57206 Mdot
.0758 .5718 Mdot
.07595 .57143 Mdot
.07612 .57125 Mdot
.0763 .57142 Mdot
.07647 .57146 Mdot
.07666 .57157 Mdot
.07686 .57197 Mdot
.07703 .57137 Mdot
.0772 .57103 Mdot
.07737 .57084 Mdot
.07753 .57046 Mdot
.07769 .57022 Mdot
.07787 .57038 Mdot
.07805 .57044 Mdot
.07823 .57051 Mdot
.07843 .5711 Mdot
.07861 .57053 Mdot
.07878 .57012 Mdot
.07895 .56997 Mdot
.07911 .56958 Mdot
.07928 .56931 Mdot
.07945 .56945 Mdot
.07963 .56953 Mdot
.07981 .56958 Mdot
.08001 .57018 Mdot
.08019 .56967 Mdot
.08036 .56921 Mdot
.08053 .56909 Mdot
.08069 .56871 Mdot
.08085 .56841 Mdot
.08103 .56852 Mdot
.08121 .56863 Mdot
.08139 .56866 Mdot
.08159 .56927 Mdot
.08177 .56885 Mdot
.08194 .56832 Mdot
.08211 .56823 Mdot
.08227 .56785 Mdot
.08243 .56753 Mdot
.0826 .56759 Mdot
.08279 .56773 Mdot
.08296 .56775 Mdot
.08316 .56833 Mdot
.08334 .56802 Mdot
.08352 .56743 Mdot
.08368 .56734 Mdot
.08385 .56696 Mdot
.08401 .56662 Mdot
.08418 .56664 Mdot
.08436 .56679 Mdot
.08453 .56681 Mdot
.08473 .56732 Mdot
.08491 .56715 Mdot
.08509 .56649 Mdot
.08526 .56638 Mdot
.08542 .56602 Mdot
.08558 .56567 Mdot
.08575 .56562 Mdot
.08593 .56579 Mdot
.0861 .56581 Mdot
.0863 .56616 Mdot
.08648 .56613 Mdot
.08666 .56546 Mdot
.08682 .5653 Mdot
.08699 .56497 Mdot
.08715 .56461 Mdot
.08732 .5645 Mdot
.08749 .56468 Mdot
.08767 .5647 Mdot
.08786 .5649 Mdot
.08805 .56505 Mdot
.08823 .56441 Mdot
.08839 .56417 Mdot
.08856 .56389 Mdot
.08872 .56353 Mdot
.08889 .56335 Mdot
.08906 .56352 Mdot
.08924 .56356 Mdot
.08943 .56368 Mdot
.08962 .56404 Mdot
.0898 .56343 Mdot
.08997 .56311 Mdot
.09014 .56291 Mdot
.0903 .56253 Mdot
.09046 .5623 Mdot
.09064 .56246 Mdot
.09081 .56252 Mdot
.091 .5626 Mdot
.0912 .56318 Mdot
.09138 .56261 Mdot
.09155 .56221 Mdot
.09172 .56205 Mdot
.09188 .56166 Mdot
.09204 .5614 Mdot
.09222 .56154 Mdot
.0924 .56162 Mdot
.09258 .56167 Mdot
.09278 .56227 Mdot
.09296 .56174 Mdot
.09313 .56129 Mdot
.0933 .56117 Mdot
.09346 .56078 Mdot
.09362 .56049 Mdot
.0938 .56061 Mdot
.09398 .56071 Mdot
.09416 .56075 Mdot
.09436 .56135 Mdot
.09454 .56091 Mdot
.09471 .5604 Mdot
.09488 .5603 Mdot
.09504 .55992 Mdot
.0952 .55961 Mdot
.09537 .55968 Mdot
.09555 .55982 Mdot
.09573 .55984 Mdot
.09593 .56043 Mdot
.09611 .5601 Mdot
.09628 .55952 Mdot
.09645 .55943 Mdot
.09661 .55905 Mdot
.09677 .55872 Mdot
.09694 .55874 Mdot
.09712 .55889 Mdot
.0973 .55891 Mdot
.0975 .55944 Mdot
.09768 .55923 Mdot
.09786 .55859 Mdot
.09802 .55849 Mdot
.09819 .55812 Mdot
.09835 .55777 Mdot
.09852 .55774 Mdot
.09869 .55791 Mdot
.09887 .55792 Mdot
.09907 .55832 Mdot
.09925 .55825 Mdot
.09943 .55758 Mdot
.09959 .55743 Mdot
.09976 .5571 Mdot
.09992 .55674 Mdot
.10009 .55665 Mdot
.10026 .55682 Mdot
.10044 .55684 Mdot
.10063 .55707 Mdot
.10082 .55717 Mdot
.10099 .55652 Mdot
.10116 .5563 Mdot
.10133 .55601 Mdot
.10149 .55565 Mdot
.10165 .55549 Mdot
.10183 .55567 Mdot
.10201 .5557 Mdot
.10219 .55584 Mdot
.10239 .55614 Mdot
.10256 .55552 Mdot
.10273 .55522 Mdot
.1029 .555 Mdot
.10306 .55463 Mdot
.10323 .55441 Mdot
.1034 .55458 Mdot
.10358 .55463 Mdot
.10376 .55471 Mdot
.10397 .55526 Mdot
.10414 .55469 Mdot
.10431 .55431 Mdot
.10448 .55414 Mdot
.10464 .55375 Mdot
.10481 .55349 Mdot
.10498 .55365 Mdot
.10516 .55372 Mdot
.10534 .55378 Mdot
.10554 .55437 Mdot
.10572 .55382 Mdot
.10589 .55339 Mdot
.10606 .55326 Mdot
.10622 .55287 Mdot
.10639 .55259 Mdot
.10656 .55272 Mdot
.10674 .55281 Mdot
.10692 .55285 Mdot
.10712 .55345 Mdot
.1073 .55299 Mdot
.10747 .55249 Mdot
.10764 .55239 Mdot
.1078 .55201 Mdot
.10796 .5517 Mdot
.10814 .55179 Mdot
.10832 .55191 Mdot
.1085 .55194 Mdot
.1087 .55254 Mdot
.10888 .55218 Mdot
.10905 .55162 Mdot
.10922 .55153 Mdot
.10938 .55115 Mdot
.10954 .55082 Mdot
.10971 .55086 Mdot
.10989 .551 Mdot
.11007 .55102 Mdot
.11027 .55157 Mdot
.11045 .55133 Mdot
.11062 .5507 Mdot
.11079 .55061 Mdot
.11095 .55023 Mdot
.11111 .54989 Mdot
.11128 .54988 Mdot
.11146 .55004 Mdot
.11164 .55005 Mdot
.11184 .55052 Mdot
.11202 .55041 Mdot
.1122 .54973 Mdot
.11236 .5496 Mdot
.11253 .54925 Mdot
.11269 .54889 Mdot
.11286 .54882 Mdot
.11303 .54899 Mdot
.11321 .54901 Mdot
.1134 .54928 Mdot
.11359 .54933 Mdot
.11376 .54867 Mdot
.11393 .54848 Mdot
.1141 .54817 Mdot
.11426 .54781 Mdot
.11442 .54767 Mdot
.1146 .54785 Mdot
.11478 .54788 Mdot
.11496 .54803 Mdot
.11516 .54827 Mdot
.11533 .54764 Mdot
.1155 .54737 Mdot
.11567 .54713 Mdot
.11583 .54676 Mdot
.11599 .54656 Mdot
.11617 .54672 Mdot
.11635 .54677 Mdot
.11653 .54687 Mdot
.11673 .54734 Mdot
.11691 .54675 Mdot
.11708 .54639 Mdot
.11725 .54621 Mdot
.11741 .54582 Mdot
.11757 .54558 Mdot
.11775 .54574 Mdot
.11793 .5458 Mdot
.11811 .54587 Mdot
.11831 .54646 Mdot
.11849 .54589 Mdot
.11866 .54547 Mdot
.11883 .54533 Mdot
.11899 .54494 Mdot
.11915 .54467 Mdot
.11933 .5448 Mdot
.11951 .54489 Mdot
.11969 .54494 Mdot
.11989 .54554 Mdot
.12007 .54505 Mdot
.12024 .54457 Mdot
.12041 .54446 Mdot
.12057 .54408 Mdot
.12073 .54378 Mdot
.1209 .54387 Mdot
.12109 .54399 Mdot
.12126 .54402 Mdot
.12147 .54463 Mdot
.12165 .54424 Mdot
.12182 .54369 Mdot
.12198 .5436 Mdot
.12215 .54322 Mdot
.12231 .54289 Mdot
.12248 .54295 Mdot
.12266 .54309 Mdot
.12284 .54311 Mdot
.12304 .54367 Mdot
.12322 .5434 Mdot
.12339 .54279 Mdot
.12356 .54269 Mdot
.12372 .54232 Mdot
.12388 .54198 Mdot
.12405 .54198 Mdot
.12423 .54213 Mdot
.12441 .54215 Mdot
.12461 .54264 Mdot
.12479 .54251 Mdot
.12496 .54183 Mdot
.12513 .54171 Mdot
.1253 .54136 Mdot
.12545 .541 Mdot
.12562 .54094 Mdot
.1258 .54111 Mdot
.12598 .54113 Mdot
.12617 .54144 Mdot
.12636 .54145 Mdot
.12653 .54078 Mdot
.1267 .5406 Mdot
.12687 .54029 Mdot
.12702 .53993 Mdot
.12719 .5398 Mdot
.12737 .53998 Mdot
.12754 .54001 Mdot
.12773 .54018 Mdot
.12792 .54038 Mdot
.1281 .53974 Mdot
.12827 .53948 Mdot
.12844 .53922 Mdot
.1286 .53886 Mdot
.12876 .53867 Mdot
.12894 .53884 Mdot
.12911 .53888 Mdot
.1293 .53899 Mdot
.1295 .5394 Mdot
.12967 .5388 Mdot
.12984 .53846 Mdot
.13001 .53827 Mdot
.13017 .53789 Mdot
.13034 .53765 Mdot
.13051 .53781 Mdot
.13069 .53788 Mdot
.13087 .53795 Mdot
.13108 .53853 Mdot
.13125 .53796 Mdot
.13142 .53755 Mdot
.13159 .5374 Mdot
.13175 .53701 Mdot
.13192 .53674 Mdot
.13209 .53688 Mdot
.13227 .53696 Mdot
.13245 .53701 Mdot
.13265 .53761 Mdot
.13283 .53711 Mdot
.13301 .53664 Mdot
.13317 .53652 Mdot
.13333 .53614 Mdot
.1335 .53584 Mdot
.13367 .53595 Mdot
.13385 .53606 Mdot
.13403 .53609 Mdot
.13423 .5367 Mdot
.13441 .53628 Mdot
.13458 .53576 Mdot
.13475 .53566 Mdot
.13491 .53528 Mdot
.13507 .53496 Mdot
.13525 .53502 Mdot
.13543 .53516 Mdot
.1356 .53518 Mdot
.1358 .53575 Mdot
.13599 .53546 Mdot
.13616 .53486 Mdot
.13633 .53477 Mdot
.13649 .53439 Mdot
.13665 .53405 Mdot
.13682 .53406 Mdot
.137 .53422 Mdot
.13717 .53423 Mdot
.13737 .53475 Mdot
.13756 .53458 Mdot
.13773 .53392 Mdot
.1379 .5338 Mdot
.13806 .53344 Mdot
.13822 .53309 Mdot
.13839 .53304 Mdot
.13857 .53321 Mdot
.13874 .53323 Mdot
.13894 .53357 Mdot
.13912 .53355 Mdot
.1393 .53288 Mdot
.13947 .53272 Mdot
.13963 .53239 Mdot
.13979 .53203 Mdot
.13996 .53192 Mdot
.14014 .5321 Mdot
.14031 .53212 Mdot
.1405 .53232 Mdot
.14069 .53247 Mdot
.14087 .53183 Mdot
.14103 .53159 Mdot
.1412 .53131 Mdot
.14136 .53095 Mdot
.14153 .53077 Mdot
.1417 .53094 Mdot
.14188 .53098 Mdot
.14207 .5311 Mdot
.14226 .53146 Mdot
.14244 .53086 Mdot
.14261 .53053 Mdot
.14278 .53033 Mdot
.14294 .52995 Mdot
.1431 .52973 Mdot
.14328 .52989 Mdot
.14346 .52995 Mdot
.14364 .53002 Mdot
.14384 .5306 Mdot
.14402 .53003 Mdot
.14419 .52963 Mdot
.14436 .52947 Mdot
.14452 .52908 Mdot
.14468 .52882 Mdot
.14486 .52897 Mdot
.14504 .52904 Mdot
.14522 .5291 Mdot
.14542 .5297 Mdot
.1456 .52917 Mdot
.14577 .52872 Mdot
.14594 .5286 Mdot
.1461 .52821 Mdot
.14626 .52792 Mdot
.14644 .52804 Mdot
.14662 .52814 Mdot
.1468 .52818 Mdot
.147 .52878 Mdot
.14718 .52834 Mdot
.14735 .52783 Mdot
.14752 .52773 Mdot
.14768 .52735 Mdot
.14784 .52703 Mdot
.14801 .52711 Mdot
.14819 .52724 Mdot
.14837 .52726 Mdot
.14857 .52785 Mdot
.14875 .52752 Mdot
.14892 .52694 Mdot
.14909 .52685 Mdot
.14925 .52647 Mdot
.14941 .52614 Mdot
.14959 .52616 Mdot
.14977 .52631 Mdot
.14994 .52633 Mdot
.15014 .52685 Mdot
.15032 .52666 Mdot
.1505 .52601 Mdot
.15066 .52591 Mdot
.15083 .52554 Mdot
.15099 .52519 Mdot
.15116 .52516 Mdot
.15134 .52532 Mdot
.15151 .52534 Mdot
.15171 .52572 Mdot
.15189 .52567 Mdot
.15207 .52499 Mdot
.15223 .52484 Mdot
.1524 .52451 Mdot
.15256 .52415 Mdot
.15273 .52405 Mdot
.1529 .52423 Mdot
.15308 .52425 Mdot
.15327 .52447 Mdot
.15346 .52459 Mdot
.15363 .52394 Mdot
.1538 .52371 Mdot
.15397 .52343 Mdot
.15413 .52307 Mdot
.1543 .5229 Mdot
.15447 .52307 Mdot
.15465 .52311 Mdot
.15484 .52324 Mdot
.15503 .52356 Mdot
.15521 .52294 Mdot
.15538 .52263 Mdot
.15554 .52242 Mdot
.1557 .52205 Mdot
.15587 .52183 Mdot
.15604 .52199 Mdot
.15622 .52205 Mdot
.15641 .52213 Mdot
.15661 .5227 Mdot
.15678 .52213 Mdot
.15696 .52173 Mdot
.15712 .52157 Mdot
.15728 .52118 Mdot
.15745 .52092 Mdot
.15762 .52107 Mdot
.1578 .52115 Mdot
.15798 .52121 Mdot
.15819 .5218 Mdot
.15837 .52126 Mdot
.15854 .52082 Mdot
.15871 .52069 Mdot
.15886 .5203 Mdot
.15903 .52002 Mdot
.1592 .52015 Mdot
.15938 .52024 Mdot
.15956 .52028 Mdot
.15976 .52088 Mdot
.15994 .52042 Mdot
.16012 .51992 Mdot
.16028 .51982 Mdot
.16044 .51944 Mdot
.16061 .51913 Mdot
.16078 .51921 Mdot
.16096 .51934 Mdot
.16114 .51937 Mdot
.16134 .51997 Mdot
.16152 .51961 Mdot
.16169 .51905 Mdot
.16186 .51896 Mdot
.16202 .51858 Mdot
.16218 .51825 Mdot
.16235 .51829 Mdot
.16253 .51843 Mdot
.16271 .51845 Mdot
.16291 .51899 Mdot
.16309 .51876 Mdot
.16327 .51813 Mdot
.16343 .51803 Mdot
.1636 .51766 Mdot
.16376 .51732 Mdot
.16393 .5173 Mdot
.1641 .51746 Mdot
.16428 .51748 Mdot
.16448 .51793 Mdot
.16466 .51783 Mdot
.16484 .51715 Mdot
.165 .51702 Mdot
.16517 .51667 Mdot
.16533 .51631 Mdot
.1655 .51624 Mdot
.16567 .51641 Mdot
.16585 .51643 Mdot
.16604 .5167 Mdot
.16623 .51675 Mdot
.16641 .51609 Mdot
.16657 .51589 Mdot
.16674 .51559 Mdot
.1669 .51523 Mdot
.16707 .51509 Mdot
.16724 .51526 Mdot
.16742 .51529 Mdot
.16761 .51545 Mdot
.1678 .51569 Mdot
.16797 .51507 Mdot
.16814 .51479 Mdot
.16831 .51455 Mdot
.16847 .51418 Mdot
.16864 .51398 Mdot
.16881 .51414 Mdot
.16899 .51419 Mdot
.16917 .51429 Mdot
.16937 .51477 Mdot
.16955 .51418 Mdot
.16972 .51382 Mdot
.16989 .51364 Mdot
.17005 .51325 Mdot
.17021 .51301 Mdot
.17039 .51316 Mdot
.17057 .51323 Mdot
.17075 .5133 Mdot
.17095 .51389 Mdot
.17113 .51333 Mdot
.1713 .5129 Mdot
.17147 .51276 Mdot
.17163 .51237 Mdot
.17179 .5121 Mdot
.17197 .51223 Mdot
.17215 .51232 Mdot
.17233 .51237 Mdot
.17253 .51297 Mdot
.17271 .51248 Mdot
.17288 .512 Mdot
.17305 .51189 Mdot
.17321 .51151 Mdot
.17337 .51121 Mdot
.17355 .5113 Mdot
.17373 .51142 Mdot
.17391 .51145 Mdot
.17411 .51206 Mdot
.17429 .51167 Mdot
.17446 .51112 Mdot
.17463 .51103 Mdot
.17479 .51065 Mdot
.17495 .51033 Mdot
.17512 .51038 Mdot
.1753 .51052 Mdot
.17548 .51054 Mdot
.17568 .51109 Mdot
.17586 .51083 Mdot
.17603 .51022 Mdot
.1762 .51012 Mdot
.17636 .50975 Mdot
.17652 .50941 Mdot
.17669 .5094 Mdot
.17687 .50956 Mdot
.17705 .50958 Mdot
.17725 .51007 Mdot
.17743 .50994 Mdot
.1776 .50926 Mdot
.17777 .50914 Mdot
.17794 .50878 Mdot
.1781 .50843 Mdot
.17826 .50837 Mdot
.17844 .50854 Mdot
.17862 .50856 Mdot
.17881 .50886 Mdot
.179 .50887 Mdot
.17917 .50821 Mdot
.17934 .50803 Mdot
.17951 .50772 Mdot
.17967 .50735 Mdot
.17983 .50723 Mdot
.18001 .5074 Mdot
.18019 .50743 Mdot
.18037 .5076 Mdot
.18057 .50781 Mdot
.18074 .50717 Mdot
.18091 .50691 Mdot
.18108 .50665 Mdot
.18124 .50629 Mdot
.1814 .5061 Mdot
.18158 .50626 Mdot
.18176 .50631 Mdot
.18194 .50641 Mdot
.18214 .50683 Mdot
.18232 .50624 Mdot
.18249 .50589 Mdot
.18266 .50571 Mdot
.18281 .50532 Mdot
.18298 .50509 Mdot
.18315 .50524 Mdot
.18333 .50531 Mdot
.18352 .50538 Mdot
.18372 .50596 Mdot
.1839 .50539 Mdot
.18407 .50498 Mdot
.18424 .50483 Mdot
.1844 .50444 Mdot
.18456 .50417 Mdot
.18474 .50431 Mdot
.18491 .50439 Mdot
.1851 .50444 Mdot
.1853 .50504 Mdot
.18548 .50454 Mdot
.18565 .50407 Mdot
.18582 .50396 Mdot
.18598 .50357 Mdot
.18614 .50328 Mdot
.18631 .50338 Mdot
.18649 .50349 Mdot
.18667 .50352 Mdot
.18687 .50413 Mdot
.18705 .50372 Mdot
.18723 .50319 Mdot
.18739 .50309 Mdot
.18755 .50271 Mdot
.18772 .50239 Mdot
.18789 .50245 Mdot
.18807 .50259 Mdot
.18825 .50261 Mdot
.18845 .50318 Mdot
.18863 .50289 Mdot
.1888 .50229 Mdot
.18897 .5022 Mdot
.18913 .50182 Mdot
.18929 .50148 Mdot
.18946 .50149 Mdot
.18964 .50165 Mdot
.18982 .50166 Mdot
.19002 .50217 Mdot
.1902 .50201 Mdot
.19037 .50135 Mdot
.19054 .50123 Mdot
.1907 .50087 Mdot
.19086 .50052 Mdot
.19103 .50047 Mdot
.19121 .50064 Mdot
.19139 .50066 Mdot
.19158 .50099 Mdot
.19177 .50098 Mdot
.19194 .5003 Mdot
.19211 .50014 Mdot
.19227 .49982 Mdot
.19243 .49946 Mdot
.1926 .49934 Mdot
.19278 .49952 Mdot
.19295 .49954 Mdot
.19314 .49974 Mdot
.19333 .4999 Mdot
.19351 .49926 Mdot
.19368 .49901 Mdot
.19385 .49874 Mdot
.194 .49838 Mdot
.19417 .4982 Mdot
.19435 .49837 Mdot
.19452 .49841 Mdot
.19471 .49852 Mdot
.19491 .49889 Mdot
.19508 .49829 Mdot
.19525 .49796 Mdot
.19542 .49776 Mdot
.19558 .49738 Mdot
.19574 .49715 Mdot
.19592 .49731 Mdot
.1961 .49737 Mdot
.19628 .49745 Mdot
.19648 .49803 Mdot
.19666 .49746 Mdot
.19683 .49706 Mdot
.197 .4969 Mdot
.19716 .49651 Mdot
.19732 .49625 Mdot
.1975 .49639 Mdot
.19768 .49647 Mdot
.19786 .49652 Mdot
.19806 .49712 Mdot
.19824 .4966 Mdot
.19841 .49615 Mdot
.19858 .49602 Mdot
.19874 .49564 Mdot
.1989 .49535 Mdot
.19908 .49546 Mdot
.19926 .49557 Mdot
.19944 .4956 Mdot
.19964 .49621 Mdot
.19982 .49577 Mdot
.19999 .49526 Mdot
.20016 .49516 Mdot
.20032 .49478 Mdot
.20048 .49446 Mdot
.20065 .49453 Mdot
.20084 .49467 Mdot
.20101 .49469 Mdot
.20121 .49527 Mdot
.20139 .49495 Mdot
.20157 .49437 Mdot
.20173 .49428 Mdot
.2019 .4939 Mdot
.20206 .49356 Mdot
.20223 .49359 Mdot
.20241 .49374 Mdot
.20258 .49375 Mdot
.20278 .49428 Mdot
.20297 .49408 Mdot
.20314 .49344 Mdot
.20331 .49333 Mdot
.20347 .49297 Mdot
.20363 .49261 Mdot
.2038 .49258 Mdot
.20398 .49275 Mdot
.20415 .49276 Mdot
.20435 .49314 Mdot
.20453 .49309 Mdot
.20471 .49241 Mdot
.20488 .49226 Mdot
.20504 .49193 Mdot
.2052 .49157 Mdot
.20537 .49147 Mdot
.20555 .49165 Mdot
.20572 .49167 Mdot
.20591 .49189 Mdot
.2061 .49201 Mdot
.20628 .49136 Mdot
.20644 .49113 Mdot
.20661 .49085 Mdot
.20677 .49049 Mdot
.20694 .49032 Mdot
.20711 .49049 Mdot
.20729 .49053 Mdot
.20748 .49066 Mdot
.20767 .49098 Mdot
.20785 .49037 Mdot
.20802 .49006 Mdot
.20819 .48985 Mdot
.20834 .48947 Mdot
.20851 .48925 Mdot
.20869 .48941 Mdot
.20886 .48947 Mdot
.20905 .48955 Mdot
.20925 .49013 Mdot
.20943 .48956 Mdot
.2096 .48916 Mdot
.20977 .489 Mdot
.20993 .48861 Mdot
.21009 .48835 Mdot
.21027 .4885 Mdot
.21044 .48858 Mdot
.21063 .48863 Mdot
.21083 .48923 Mdot
.21101 .48869 Mdot
.21118 .48825 Mdot
.21135 .48812 Mdot
.21151 .48773 Mdot
.21167 .48745 Mdot
.21185 .48758 Mdot
.21203 .48767 Mdot
.21221 .48771 Mdot
.21241 .48831 Mdot
.21259 .48785 Mdot
.21276 .48735 Mdot
.21293 .48725 Mdot
.21309 .48687 Mdot
.21325 .48656 Mdot
.21342 .48664 Mdot
.2136 .48677 Mdot
.21378 .4868 Mdot
.21398 .4874 Mdot
.21416 .48704 Mdot
.21434 .48648 Mdot
.2145 .48639 Mdot
.21466 .48601 Mdot
.21483 .48568 Mdot
.215 .48571 Mdot
.21518 .48586 Mdot
.21535 .48588 Mdot
.21555 .48642 Mdot
.21573 .48619 Mdot
.21591 .48556 Mdot
.21608 .48546 Mdot
.21624 .48509 Mdot
.2164 .48474 Mdot
.21657 .48472 Mdot
.21675 .48489 Mdot
.21692 .4849 Mdot
.21712 .48535 Mdot
.2173 .48525 Mdot
.21748 .48457 Mdot
.21765 .48444 Mdot
.21781 .48409 Mdot
.21797 .48373 Mdot
.21814 .48366 Mdot
.21832 .48383 Mdot
.21849 .48385 Mdot
.21868 .48411 Mdot
.21887 .48417 Mdot
.21905 .48351 Mdot
.21921 .48331 Mdot
.21938 .48301 Mdot
.21954 .48265 Mdot
.21971 .48251 Mdot
.21988 .48268 Mdot
.22006 .48271 Mdot
.22025 .48286 Mdot
.22044 .48312 Mdot
.22062 .48249 Mdot
.22079 .48221 Mdot
.22095 .48198 Mdot
.22111 .48161 Mdot
.22128 .4814 Mdot
.22145 .48157 Mdot
.22163 .48162 Mdot
.22182 .48171 Mdot
.22202 .4822 Mdot
.22219 .48162 Mdot
.22236 .48125 Mdot
.22253 .48107 Mdot
.22269 .48069 Mdot
.22286 .48044 Mdot
.22303 .48059 Mdot
.22321 .48066 Mdot
.22339 .48073 Mdot
.22359 .48132 Mdot
.22377 .48076 Mdot
.22394 .48033 Mdot
.22411 .4802 Mdot
.22427 .47981 Mdot
.22444 .47953 Mdot
.22461 .47966 Mdot
.22479 .47975 Mdot
.22497 .4798 Mdot
.22517 .4804 Mdot
.22535 .47992 Mdot
.22552 .47943 Mdot
.22569 .47932 Mdot
.22585 .47894 Mdot
.22601 .47864 Mdot
.22619 .47873 Mdot
.22637 .47885 Mdot
.22655 .47888 Mdot
.22675 .47949 Mdot
.22693 .47911 Mdot
.2271 .47856 Mdot
.22727 .47846 Mdot
.22743 .47808 Mdot
.22759 .47776 Mdot
.22776 .47781 Mdot
.22794 .47795 Mdot
.22812 .47797 Mdot
.22832 .47852 Mdot
.2285 .47826 Mdot
.22867 .47765 Mdot
.22884 .47755 Mdot
.229 .47718 Mdot
.22917 .47684 Mdot
.22933 .47683 Mdot
.22951 .47699 Mdot
.22969 .477 Mdot
.22989 .4775 Mdot
.23007 .47737 Mdot
.23025 .47669 Mdot
.23041 .47657 Mdot
.23058 .47621 Mdot
.23074 .47586 Mdot
.23091 .47579 Mdot
.23108 .47597 Mdot
.23126 .47598 Mdot
.23145 .47628 Mdot
.23164 .4763 Mdot
.23182 .47564 Mdot
.23198 .47545 Mdot
.23215 .47514 Mdot
.23231 .47478 Mdot
.23247 .47465 Mdot
.23265 .47483 Mdot
.23283 .47486 Mdot
.23302 .47503 Mdot
.23321 .47523 Mdot
.23338 .4746 Mdot
.23355 .47434 Mdot
.23372 .47408 Mdot
.23388 .47372 Mdot
.23405 .47352 Mdot
.23422 .47369 Mdot
.2344 .47374 Mdot
.23458 .47384 Mdot
.23478 .47427 Mdot
.23496 .47367 Mdot
.23513 .47333 Mdot
.2353 .47314 Mdot
.23546 .47276 Mdot
.23562 .47252 Mdot
.2358 .47268 Mdot
.23597 .47274 Mdot
.23616 .47281 Mdot
.23636 .4734 Mdot
.23654 .47282 Mdot
.23671 .47241 Mdot
.23688 .47226 Mdot
.23704 .47187 Mdot
.2372 .4716 Mdot
.23738 .47174 Mdot
.23756 .47182 Mdot
.23774 .47187 Mdot
.23794 .47247 Mdot
.23812 .47197 Mdot
.23829 .4715 Mdot
.23846 .47139 Mdot
.23862 .471 Mdot
.23878 .47071 Mdot
.23895 .47081 Mdot
.23914 .47092 Mdot
.23932 .47096 Mdot
.23952 .47156 Mdot
.2397 .47115 Mdot
.23987 .47062 Mdot
.24004 .47053 Mdot
.2402 .47014 Mdot
.24036 .46982 Mdot
.24053 .46988 Mdot
.24071 .47002 Mdot
.24089 .47004 Mdot
.24109 .47061 Mdot
.24127 .47032 Mdot
.24144 .46972 Mdot
.24161 .46963 Mdot
.24177 .46925 Mdot
.24193 .46891 Mdot
.2421 .46892 Mdot
.24228 .46908 Mdot
.24246 .46909 Mdot
.24266 .4696 Mdot
.24284 .46944 Mdot
.24301 .46878 Mdot
.24318 .46866 Mdot
.24335 .4683 Mdot
.2435 .46795 Mdot
.24367 .4679 Mdot
.24385 .46807 Mdot
.24403 .46808 Mdot
.24422 .46842 Mdot
.24441 .4684 Mdot
.24458 .46773 Mdot
.24475 .46757 Mdot
.24492 .46725 Mdot
.24508 .46689 Mdot
.24524 .46677 Mdot
.24542 .46694 Mdot
.2456 .46697 Mdot
.24578 .46716 Mdot
.24597 .46733 Mdot
.24615 .46669 Mdot
.24632 .46644 Mdot
.24649 .46617 Mdot
.24665 .46581 Mdot
.24681 .46562 Mdot
.24699 .46579 Mdot
.24717 .46583 Mdot
.24735 .46595 Mdot
.24755 .46632 Mdot
.24772 .46572 Mdot
.24789 .46539 Mdot
.24806 .4652 Mdot
.24822 .46481 Mdot
.24839 .46458 Mdot
.24856 .46474 Mdot
.24874 .4648 Mdot
.24892 .46488 Mdot
.24913 .46546 Mdot
.2493 .46489 Mdot
.24947 .46449 Mdot
.24964 .46433 Mdot
.2498 .46394 Mdot
.24997 .46368 Mdot
.25014 .46382 Mdot
.25032 .4639 Mdot
.2505 .46395 Mdot
.2507 .46455 Mdot
.25088 .46403 Mdot
.25106 .46357 Mdot
.25122 .46345 Mdot
.25138 .46307 Mdot
.25155 .46278 Mdot
.25172 .46289 Mdot
.2519 .463 Mdot
.25208 .46303 Mdot
.25228 .46363 Mdot
.25246 .4632 Mdot
.25263 .46269 Mdot
.2528 .46259 Mdot
.25296 .46221 Mdot
.25312 .46189 Mdot
.2533 .46196 Mdot
.25348 .4621 Mdot
.25366 .46212 Mdot
.25386 .4627 Mdot
.25404 .46238 Mdot
.25421 .4618 Mdot
.25438 .46171 Mdot
.25454 .46133 Mdot
.2547 .46099 Mdot
.25487 .46101 Mdot
.25505 .46117 Mdot
.25523 .46118 Mdot
.25543 .4617 Mdot
.25561 .46151 Mdot
.25578 .46086 Mdot
.25595 .46076 Mdot
.25611 .46039 Mdot
.25627 .46004 Mdot
.25644 .46 Mdot
.25662 .46017 Mdot
.2568 .46019 Mdot
.25699 .46056 Mdot
.25718 .46051 Mdot
.25735 .45984 Mdot
.25752 .45969 Mdot
.25768 .45936 Mdot
.25784 .459 Mdot
.25801 .4589 Mdot
.25819 .45907 Mdot
.25836 .45909 Mdot
.25855 .45931 Mdot
.25874 .45943 Mdot
.25892 .45878 Mdot
.25909 .45856 Mdot
.25926 .45827 Mdot
.25941 .45791 Mdot
.25958 .45774 Mdot
.25976 .45792 Mdot
.25993 .45795 Mdot
.26012 .45808 Mdot
.26031 .45841 Mdot
.26049 .4578 Mdot
.26066 .45748 Mdot
.26083 .45728 Mdot
.26099 .4569 Mdot
.26115 .45668 Mdot
.26133 .45684 Mdot
.26151 .4569 Mdot
.26169 .45698 Mdot
.26189 .45755 Mdot
.26207 .45698 Mdot
.26224 .45659 Mdot
.26241 .45643 Mdot
.26257 .45603 Mdot
.26273 .45578 Mdot
.26291 .45593 Mdot
.26309 .456 Mdot
.26327 .45606 Mdot
.26347 .45666 Mdot
.26365 .45611 Mdot
.26382 .45567 Mdot
.26399 .45555 Mdot
.26415 .45516 Mdot
.26431 .45487 Mdot
.26449 .455 Mdot
.26467 .45509 Mdot
.26485 .45514 Mdot
.26505 .45574 Mdot
.26523 .45528 Mdot
.2654 .45478 Mdot
.26557 .45468 Mdot
.26573 .4543 Mdot
.26589 .45399 Mdot
.26606 .45407 Mdot
.26625 .4542 Mdot
.26642 .45422 Mdot
.26662 .45482 Mdot
.2668 .45447 Mdot
.26698 .4539 Mdot
.26714 .45381 Mdot
.26731 .45343 Mdot
.26747 .4531 Mdot
.26764 .45314 Mdot
.26782 .45328 Mdot
.268 .4533 Mdot
.2682 .45384 Mdot
.26838 .45361 Mdot
.26855 .45298 Mdot
.26872 .45288 Mdot
.26888 .45251 Mdot
.26904 .45216 Mdot
.26921 .45214 Mdot
.26939 .45231 Mdot
.26956 .45232 Mdot
.26976 .45276 Mdot
.26995 .45267 Mdot
.27012 .45199 Mdot
.27029 .45185 Mdot
.27045 .45151 Mdot
.27061 .45115 Mdot
.27078 .45107 Mdot
.27096 .45124 Mdot
.27113 .45126 Mdot
.27133 .45152 Mdot
.27151 .45159 Mdot
.27169 .45093 Mdot
.27186 .45073 Mdot
.27202 .45043 Mdot
.27218 .45007 Mdot
.27235 .44992 Mdot
.27253 .45009 Mdot
.2727 .45013 Mdot
.27289 .45028 Mdot
.27308 .45054 Mdot
.27326 .44991 Mdot
.27343 .44963 Mdot
.2736 .4494 Mdot
.27375 .44903 Mdot
.27392 .44882 Mdot
.2741 .44898 Mdot
.27427 .44904 Mdot
.27446 .44913 Mdot
.27466 .44963 Mdot
.27484 .44905 Mdot
.27501 .44868 Mdot
.27517 .4485 Mdot
.27533 .44811 Mdot
.2755 .44787 Mdot
.27567 .44802 Mdot
.27585 .44809 Mdot
.27603 .44815 Mdot
.27624 .44874 Mdot
.27642 .44819 Mdot
.27659 .44776 Mdot
.27675 .44762 Mdot
.27691 .44723 Mdot
.27708 .44696 Mdot
.27725 .44709 Mdot
.27743 .44718 Mdot
.27761 .44722 Mdot
.27781 .44782 Mdot
.27799 .44735 Mdot
.27817 .44686 Mdot
.27833 .44675 Mdot
.27849 .44637 Mdot
.27866 .44607 Mdot
.27883 .44616 Mdot
.27901 .44628 Mdot
.27919 .44631 Mdot
.27939 .44691 Mdot
.27957 .44654 Mdot
.27974 .44599 Mdot
.27991 .44589 Mdot
.28007 .44551 Mdot
.28023 .44519 Mdot
.2804 .44523 Mdot
.28059 .44538 Mdot
.28076 .44539 Mdot
.28096 .44595 Mdot
.28114 .44569 Mdot
.28132 .44507 Mdot
.28148 .44498 Mdot
.28165 .44461 Mdot
.28181 .44426 Mdot
.28198 .44426 Mdot
.28216 .44442 Mdot
.28233 .44443 Mdot
.28253 .44492 Mdot
.28271 .4448 Mdot
.28289 .44412 Mdot
.28306 .44399 Mdot
.28322 .44364 Mdot
.28338 .44328 Mdot
.28355 .44322 Mdot
.28373 .44339 Mdot
.2839 .44341 Mdot
.2841 .4437 Mdot
.28428 .44373 Mdot
.28446 .44306 Mdot
.28462 .44288 Mdot
.28479 .44257 Mdot
.28495 .44221 Mdot
.28512 .44207 Mdot
.28529 .44225 Mdot
.28547 .44228 Mdot
.28566 .44245 Mdot
.28585 .44266 Mdot
.28603 .44203 Mdot
.28619 .44176 Mdot
.28636 .44151 Mdot
.28652 .44114 Mdot
.28669 .44095 Mdot
.28686 .44112 Mdot
.28704 .44116 Mdot
.28723 .44126 Mdot
.28742 .4417 Mdot
.2876 .44111 Mdot
.28777 .44075 Mdot
.28794 .44057 Mdot
.2881 .44019 Mdot
.28826 .43995 Mdot
.28844 .4401 Mdot
.28862 .44017 Mdot
.2888 .44024 Mdot
.289 .44082 Mdot
.28918 .44025 Mdot
.28935 .43984 Mdot
.28952 .43969 Mdot
.28968 .4393 Mdot
.28984 .43903 Mdot
.29002 .43917 Mdot
.2902 .43925 Mdot
.29038 .4393 Mdot
.29058 .4399 Mdot
.29076 .4394 Mdot
.29093 .43893 Mdot
.2911 .43882 Mdot
.29126 .43844 Mdot
.29142 .43814 Mdot
.2916 .43824 Mdot
.29178 .43835 Mdot
.29196 .43839 Mdot
.29216 .43899 Mdot
.29234 .43859 Mdot
.29251 .43805 Mdot
.29268 .43796 Mdot
.29284 .43758 Mdot
.293 .43725 Mdot
.29317 .43731 Mdot
.29335 .43745 Mdot
.29353 .43747 Mdot
.29373 .43804 Mdot
.29391 .43775 Mdot
.29408 .43715 Mdot
.29425 .43706 Mdot
.29441 .43668 Mdot
.29457 .43634 Mdot
.29474 .43635 Mdot
.29492 .43651 Mdot
.2951 .43652 Mdot
.2953 .43703 Mdot
.29548 .43687 Mdot
.29566 .4362 Mdot
.29582 .43609 Mdot
.29599 .43573 Mdot
.29615 .43538 Mdot
.29632 .43532 Mdot
.29649 .43549 Mdot
.29667 .43551 Mdot
.29686 .43584 Mdot
.29705 .43583 Mdot
.29722 .43516 Mdot
.29739 .43499 Mdot
.29756 .43467 Mdot
.29772 .43431 Mdot
.29788 .43419 Mdot
.29806 .43437 Mdot
.29824 .43439 Mdot
.29843 .43458 Mdot
.29862 .43476 Mdot
.29879 .43412 Mdot
.29896 .43387 Mdot
.29913 .43359 Mdot
.29929 .43323 Mdot
.29945 .43305 Mdot
.29963 .43322 Mdot
.29981 .43326 Mdot
.29999 .43337 Mdot
.30019 .43375 Mdot
.30037 .43315 Mdot
.30054 .43282 Mdot
.30071 .43263 Mdot
.30086 .43225 Mdot
.30103 .43201 Mdot
.3012 .43217 Mdot
.30138 .43223 Mdot
.30157 .43231 Mdot
.30177 .43289 Mdot
.30195 .43232 Mdot
.30212 .43191 Mdot
.30229 .43176 Mdot
.30245 .43137 Mdot
.30261 .4311 Mdot
.30279 .43125 Mdot
.30296 .43133 Mdot
.30315 .43138 Mdot
.30335 .43198 Mdot
.30353 .43146 Mdot
.3037 .431 Mdot
.30387 .43088 Mdot
.30403 .4305 Mdot
.30419 .43021 Mdot
.30436 .43032 Mdot
.30454 .43042 Mdot
.30472 .43046 Mdot
.30492 .43106 Mdot
.3051 .43064 Mdot
.30528 .43012 Mdot
.30544 .43002 Mdot
.3056 .42964 Mdot
.30577 .42932 Mdot
.30594 .42939 Mdot
.30612 .42952 Mdot
.3063 .42955 Mdot
.3065 .43013 Mdot
.30668 .42981 Mdot
.30685 .42923 Mdot
.30702 .42914 Mdot
.30718 .42876 Mdot
.30734 .42842 Mdot
.30751 .42844 Mdot
.30769 .42859 Mdot
.30787 .42861 Mdot
.30807 .42913 Mdot
.30825 .42894 Mdot
.30842 .42829 Mdot
.30859 .42818 Mdot
.30875 .42782 Mdot
.30891 .42747 Mdot
.30908 .42743 Mdot
.30926 .4276 Mdot
.30944 .42761 Mdot
.30963 .42798 Mdot
.30982 .42794 Mdot
.30999 .42726 Mdot
.31016 .42711 Mdot
.31033 .42678 Mdot
.31048 .42642 Mdot
.31065 .42632 Mdot
.31083 .42649 Mdot
.311 .42651 Mdot
.3112 .42673 Mdot
.31138 .42686 Mdot
.31156 .42621 Mdot
.31173 .42598 Mdot
.3119 .4257 Mdot
.31205 .42534 Mdot
.31222 .42516 Mdot
.3124 .42534 Mdot
.31257 .42537 Mdot
.31276 .4255 Mdot
.31296 .42583 Mdot
.31313 .42522 Mdot
.3133 .42491 Mdot
.31347 .4247 Mdot
.31363 .42433 Mdot
.31379 .4241 Mdot
.31397 .42426 Mdot
.31415 .42432 Mdot
.31433 .4244 Mdot
.31453 .42498 Mdot
.31471 .42441 Mdot
.31488 .42401 Mdot
.31505 .42385 Mdot
.31521 .42346 Mdot
.31537 .4232 Mdot
.31555 .42335 Mdot
.31573 .42342 Mdot
.31591 .42348 Mdot
.31611 .42408 Mdot
.31629 .42354 Mdot
.31646 .4231 Mdot
.31663 .42297 Mdot
.31679 .42258 Mdot
.31695 .4223 Mdot
.31713 .42242 Mdot
.31731 .42252 Mdot
.31749 .42256 Mdot
.31769 .42316 Mdot
.31787 .42271 Mdot
.31804 .4222 Mdot
.31821 .4221 Mdot
.31837 .42172 Mdot
.31853 .42141 Mdot
.31871 .42149 Mdot
.31889 .42162 Mdot
.31907 .42165 Mdot
.31927 .42224 Mdot
.31945 .4219 Mdot
.31962 .42132 Mdot
.31979 .42123 Mdot
.31995 .42085 Mdot
.32011 .42052 Mdot
.32028 .42055 Mdot
.32046 .4207 Mdot
.32064 .42072 Mdot
.32084 .42126 Mdot
.32102 .42104 Mdot
.32119 .4204 Mdot
.32136 .4203 Mdot
.32152 .41993 Mdot
.32168 .41958 Mdot
.32185 .41956 Mdot
.32203 .41973 Mdot
.32221 .41974 Mdot
.3224 .42016 Mdot
.32259 .42008 Mdot
.32276 .4194 Mdot
.32293 .41926 Mdot
.32309 .41892 Mdot
.32325 .41856 Mdot
.32342 .41848 Mdot
.3236 .41865 Mdot
.32377 .41867 Mdot
.32397 .41892 Mdot
.32415 .419 Mdot
.32433 .41834 Mdot
.3245 .41814 Mdot
.32467 .41784 Mdot
.32482 .41748 Mdot
.32499 .41733 Mdot
.32517 .4175 Mdot
.32534 .41753 Mdot
.32553 .41768 Mdot
.32572 .41795 Mdot
.3259 .41733 Mdot
.32607 .41704 Mdot
.32624 .41682 Mdot
.3264 .41644 Mdot
.32656 .41623 Mdot
.32674 .4164 Mdot
.32691 .41645 Mdot
.3271 .41654 Mdot
.3273 .41706 Mdot
.32748 .41648 Mdot
.32765 .4161 Mdot
.32782 .41593 Mdot
.32798 .41554 Mdot
.32814 .41529 Mdot
.32832 .41544 Mdot
.32849 .41551 Mdot
.32868 .41558 Mdot
.32888 .41617 Mdot
.32906 .41562 Mdot
.32923 .41519 Mdot
.3294 .41505 Mdot
.32956 .41466 Mdot
.32972 .41438 Mdot
.3299 .41452 Mdot
.33007 .41461 Mdot
.33026 .41465 Mdot
.33046 .41525 Mdot
.33064 .41478 Mdot
.33081 .41429 Mdot
.33098 .41418 Mdot
.33114 .4138 Mdot
.3313 .4135 Mdot
.33147 .41358 Mdot
.33165 .41371 Mdot
.33183 .41374 Mdot
.33203 .41434 Mdot
.33221 .41397 Mdot
.33239 .41341 Mdot
.33255 .41332 Mdot
.33271 .41294 Mdot
.33288 .41261 Mdot
.33305 .41266 Mdot
.33323 .4128 Mdot
.3334 .41282 Mdot
.3336 .41337 Mdot
.33379 .41312 Mdot
.33396 .4125 Mdot
.33413 .41241 Mdot
.33429 .41203 Mdot
.33445 .41169 Mdot
.33462 .41168 Mdot
.3348 .41184 Mdot
.33497 .41185 Mdot
.33517 .41234 Mdot
.33536 .41222 Mdot
.33553 .41154 Mdot
.3357 .41141 Mdot
.33586 .41106 Mdot
.33602 .4107 Mdot
.33619 .41064 Mdot
.33637 .41081 Mdot
.33654 .41083 Mdot
.33674 .41112 Mdot
.33692 .41115 Mdot
.3371 .41048 Mdot
.33727 .4103 Mdot
.33743 .40999 Mdot
.33759 .40963 Mdot
.33776 .40949 Mdot
.33794 .40967 Mdot
.33811 .4097 Mdot
.3383 .40986 Mdot
.33849 .41008 Mdot
.33867 .40945 Mdot
.33884 .40918 Mdot
.33901 .40893 Mdot
.33916 .40857 Mdot
.33933 .40837 Mdot
.33951 .40854 Mdot
.33968 .40858 Mdot
.33987 .40868 Mdot
.34007 .40913 Mdot
.34024 .40854 Mdot
.34041 .40818 Mdot
.34058 .408 Mdot
.34074 .40762 Mdot
.34091 .40738 Mdot
.34108 .40753 Mdot
.34126 .4076 Mdot
.34144 .40767 Mdot
.34164 .40825 Mdot
.34182 .40768 Mdot
.34199 .40727 Mdot
.34216 .40712 Mdot
.34232 .40673 Mdot
.34249 .40646 Mdot
.34266 .4066 Mdot
.34284 .40668 Mdot
.34302 .40673 Mdot
.34322 .40733 Mdot
.3434 .40684 Mdot
.34357 .40636 Mdot
.34374 .40625 Mdot
.3439 .40587 Mdot
.34406 .40557 Mdot
.34424 .40567 Mdot
.34442 .40578 Mdot
.3446 .40582 Mdot
.3448 .40642 Mdot
.34498 .40602 Mdot
.34515 .40548 Mdot
.34532 .40539 Mdot
.34548 .40501 Mdot
.34564 .40469 Mdot
.34581 .40474 Mdot
.34599 .40488 Mdot
.34617 .4049 Mdot
.34637 .40547 Mdot
.34655 .40519 Mdot
.34673 .40458 Mdot
.34689 .40449 Mdot
.34706 .40411 Mdot
.34722 .40377 Mdot
.34739 .40378 Mdot
.34756 .40393 Mdot
.34774 .40395 Mdot
.34794 .40445 Mdot
.34812 .4043 Mdot
.3483 .40363 Mdot
.34846 .40351 Mdot
.34863 .40316 Mdot
.34879 .4028 Mdot
.34896 .40275 Mdot
.34914 .40292 Mdot
.34931 .40293 Mdot
.34951 .40326 Mdot
.34969 .40325 Mdot
.34987 .40258 Mdot
.35003 .40241 Mdot
.3502 .4021 Mdot
.35036 .40173 Mdot
.35053 .40161 Mdot
.3507 .40179 Mdot
.35088 .40182 Mdot
.35107 .402 Mdot
.35126 .40218 Mdot
.35143 .40154 Mdot
.3516 .40129 Mdot
.35177 .40102 Mdot
.35193 .40066 Mdot
.3521 .40047 Mdot
.35227 .40064 Mdot
.35245 .40069 Mdot
.35263 .4008 Mdot
.35283 .40119 Mdot
.35301 .40059 Mdot
.35318 .40025 Mdot
.35335 .40006 Mdot
.35351 .39968 Mdot
.35367 .39945 Mdot
.35385 .3996 Mdot
.35402 .39967 Mdot
.35421 .39974 Mdot
.35441 .40032 Mdot
.35459 .39975 Mdot
.35476 .39934 Mdot
.35493 .39919 Mdot
.35509 .3988 Mdot
.35525 .39853 Mdot
.35543 .39868 Mdot
.35561 .39876 Mdot
.35579 .39881 Mdot
.35599 .39941 Mdot
.35617 .39889 Mdot
.35634 .39843 Mdot
.35651 .39832 Mdot
.35667 .39793 Mdot
.35683 .39764 Mdot
.35701 .39775 Mdot
.35719 .39785 Mdot
.35737 .39789 Mdot
.35757 .39849 Mdot
.35775 .39807 Mdot
.35792 .39755 Mdot
.35809 .39745 Mdot
.35825 .39707 Mdot
.35841 .39675 Mdot
.35858 .39682 Mdot
.35876 .39695 Mdot
.35894 .39698 Mdot
.35914 .39755 Mdot
.35932 .39725 Mdot
.35949 .39666 Mdot
.35966 .39656 Mdot
.35982 .39619 Mdot
.35998 .39585 Mdot
.36015 .39586 Mdot
.36033 .39602 Mdot
.36051 .39603 Mdot
.36071 .39655 Mdot
.36089 .39637 Mdot
.36106 .39572 Mdot
.36123 .39561 Mdot
.3614 .39525 Mdot
.36156 .39489 Mdot
.36172 .39485 Mdot
.3619 .39502 Mdot
.36208 .39503 Mdot
.36227 .39539 Mdot
.36246 .39536 Mdot
.36263 .39468 Mdot
.3628 .39453 Mdot
.36297 .3942 Mdot
.36313 .39384 Mdot
.36329 .39373 Mdot
.36347 .39391 Mdot
.36365 .39393 Mdot
.36384 .39414 Mdot
.36403 .39428 Mdot
.3642 .39363 Mdot
.36437 .3934 Mdot
.36454 .39312 Mdot
.3647 .39276 Mdot
.36486 .39258 Mdot
.36504 .39275 Mdot
.36522 .39279 Mdot
.3654 .39292 Mdot
.3656 .39326 Mdot
.36577 .39265 Mdot
.36594 .39233 Mdot
.36611 .39213 Mdot
.36627 .39175 Mdot
.36644 .39153 Mdot
.36661 .39169 Mdot
.36679 .39175 Mdot
.36697 .39182 Mdot
.36717 .3924 Mdot
.36735 .39183 Mdot
.36752 .39143 Mdot
.36769 .39128 Mdot
.36785 .39088 Mdot
.36802 .39062 Mdot
.36819 .39077 Mdot
.36837 .39085 Mdot
.36855 .3909 Mdot
.36875 .3915 Mdot
.36893 .39097 Mdot
.3691 .39052 Mdot
.36927 .3904 Mdot
.36943 .39001 Mdot
.3696 .38972 Mdot
.36977 .38984 Mdot
.36995 .38994 Mdot
.37013 .38998 Mdot
.37033 .39058 Mdot
.37051 .39014 Mdot
.37068 .38963 Mdot
.37085 .38953 Mdot
.37101 .38915 Mdot
.37117 .38884 Mdot
.37135 .38891 Mdot
.37153 .38904 Mdot
.37171 .38907 Mdot
.37191 .38966 Mdot
.37209 .38932 Mdot
.37226 .38875 Mdot
.37243 .38866 Mdot
.37259 .38828 Mdot
.37275 .38794 Mdot
.37292 .38797 Mdot
.3731 .38812 Mdot
.37328 .38814 Mdot
.37348 .38867 Mdot
.37366 .38846 Mdot
.37383 .38782 Mdot
.374 .38772 Mdot
.37416 .38735 Mdot
.37432 .387 Mdot
.37449 .38698 Mdot
.37467 .38714 Mdot
.37485 .38715 Mdot
.37504 .38756 Mdot
.37523 .38749 Mdot
.3754 .38681 Mdot
.37557 .38667 Mdot
.37574 .38633 Mdot
.37589 .38597 Mdot
.37606 .38588 Mdot
.37624 .38606 Mdot
.37642 .38608 Mdot
.37661 .38632 Mdot
.37679 .38641 Mdot
.37697 .38576 Mdot
.37714 .38554 Mdot
.37731 .38525 Mdot
.37746 .38489 Mdot
.37763 .38473 Mdot
.37781 .38491 Mdot
.37798 .38494 Mdot
.37817 .38508 Mdot
.37836 .38537 Mdot
.37854 .38475 Mdot
.37871 .38445 Mdot
.37888 .38423 Mdot
.37904 .38386 Mdot
.3792 .38365 Mdot
.37938 .38381 Mdot
.37956 .38386 Mdot
.37974 .38395 Mdot
.37994 .38449 Mdot
.38012 .38391 Mdot
.38029 .38353 Mdot
.38046 .38336 Mdot
.38062 .38297 Mdot
.38078 .38272 Mdot
.38096 .38287 Mdot
.38114 .38294 Mdot
.38132 .383 Mdot
.38152 .3836 Mdot
.3817 .38305 Mdot
.38187 .38261 Mdot
.38204 .38248 Mdot
.3822 .38209 Mdot
.38236 .38181 Mdot
.38254 .38194 Mdot
.38272 .38203 Mdot
.3829 .38208 Mdot
.3831 .38268 Mdot
.38328 .38221 Mdot
.38345 .38172 Mdot
.38362 .38161 Mdot
.38378 .38123 Mdot
.38394 .38092 Mdot
.38411 .38101 Mdot
.3843 .38114 Mdot
.38447 .38117 Mdot
.38467 .38177 Mdot
.38485 .3814 Mdot
.38503 .38084 Mdot
.38519 .38075 Mdot
.38536 .38037 Mdot
.38552 .38004 Mdot
.38569 .38009 Mdot
.38587 .38023 Mdot
.38605 .38025 Mdot
.38625 .3808 Mdot
.38643 .38055 Mdot
.3866 .37993 Mdot
.38677 .37983 Mdot
.38693 .37946 Mdot
.38709 .37912 Mdot
.38726 .3791 Mdot
.38744 .37927 Mdot
.38762 .37928 Mdot
.38781 .37975 Mdot
.388 .37964 Mdot
.38817 .37896 Mdot
.38834 .37883 Mdot
.3885 .37848 Mdot
.38866 .37812 Mdot
.38883 .37805 Mdot
.38901 .37822 Mdot
.38919 .37824 Mdot
.38938 .37852 Mdot
.38956 .37856 Mdot
.38974 .3779 Mdot
.38991 .37771 Mdot
.39008 .37741 Mdot
.39023 .37704 Mdot
.3904 .37691 Mdot
.39058 .37708 Mdot
.39075 .37711 Mdot
.39094 .37727 Mdot
.39113 .3775 Mdot
.39131 .37687 Mdot
.39148 .3766 Mdot
.39165 .37636 Mdot
.39181 .37599 Mdot
.39197 .37579 Mdot
.39215 .37596 Mdot
.39232 .376 Mdot
.39251 .3761 Mdot
.39271 .37656 Mdot
.39288 .37597 Mdot
.39305 .37561 Mdot
.39322 .37543 Mdot
.39338 .37505 Mdot
.39355 .37481 Mdot
.39372 .37496 Mdot
.3939 .37503 Mdot
.39408 .37509 Mdot
.39429 .37568 Mdot
.39446 .37512 Mdot
.39464 .3747 Mdot
.3948 .37456 Mdot
.39496 .37417 Mdot
.39513 .37389 Mdot
.3953 .37403 Mdot
.39548 .37411 Mdot
.39566 .37416 Mdot
.39586 .37476 Mdot
.39604 .37427 Mdot
.39622 .37379 Mdot
.39638 .37368 Mdot
.39654 .3733 Mdot
.39671 .373 Mdot
.39688 .3731 Mdot
.39706 .37322 Mdot
.39724 .37325 Mdot
.39744 .37385 Mdot
.39762 .37346 Mdot
.39779 .37292 Mdot
.39796 .37282 Mdot
.39812 .37244 Mdot
.39828 .37212 Mdot
.39846 .37217 Mdot
.39864 .37231 Mdot
.39881 .37233 Mdot
.39901 .37289 Mdot
.39919 .37262 Mdot
.39937 .37201 Mdot
.39953 .37192 Mdot
.3997 .37154 Mdot
.39986 .3712 Mdot
.40003 .3712 Mdot
.40021 .37136 Mdot
.40038 .37137 Mdot
.40058 .37187 Mdot
.40076 .37173 Mdot
.40094 .37106 Mdot
.40111 .37094 Mdot
.40127 .37058 Mdot
.40143 .37023 Mdot
.4016 .37017 Mdot
.40178 .37034 Mdot
.40195 .37036 Mdot
.40215 .37067 Mdot
.40233 .37068 Mdot
.40251 .37001 Mdot
.40267 .36983 Mdot
.40284 .36952 Mdot
.403 .36916 Mdot
.40317 .36903 Mdot
.40334 .36921 Mdot
.40352 .36924 Mdot
.40371 .36942 Mdot
.4039 .36961 Mdot
.40408 .36897 Mdot
.40424 .36871 Mdot
.40441 .36845 Mdot
.40457 .36809 Mdot
.40474 .3679 Mdot
.40491 .36807 Mdot
.40509 .36811 Mdot
.40528 .36822 Mdot
.40547 .36862 Mdot
.40565 .36802 Mdot
.40582 .36768 Mdot
.40599 .3675 Mdot
.40615 .36711 Mdot
.40631 .36688 Mdot
.40649 .36704 Mdot
.40667 .3671 Mdot
.40685 .36717 Mdot
.40705 .36775 Mdot
.40723 .36718 Mdot
.4074 .36677 Mdot
.40757 .36662 Mdot
.40773 .36623 Mdot
.40789 .36596 Mdot
.40807 .36611 Mdot
.40825 .36619 Mdot
.40843 .36624 Mdot
.40863 .36684 Mdot
.40881 .36633 Mdot
.40898 .36586 Mdot
.40915 .36575 Mdot
.40931 .36536 Mdot
.40947 .36507 Mdot
.40965 .36518 Mdot
.40983 .36528 Mdot
.41001 .36532 Mdot
.41021 .36592 Mdot
.41039 .36551 Mdot
.41056 .36498 Mdot
.41073 .36488 Mdot
.41089 .3645 Mdot
.41105 .36418 Mdot
.41122 .36425 Mdot
.4114 .36438 Mdot
.41158 .3644 Mdot
.41178 .36498 Mdot
.41196 .36468 Mdot
.41213 .36408 Mdot
.4123 .36399 Mdot
.41246 .36362 Mdot
.41262 .36328 Mdot
.41279 .36329 Mdot
.41297 .36345 Mdot
.41315 .36346 Mdot
.41335 .36397 Mdot
.41353 .3638 Mdot
.41371 .36314 Mdot
.41387 .36303 Mdot
.41404 .36267 Mdot
.4142 .36232 Mdot
.41437 .36227 Mdot
.41454 .36244 Mdot
.41472 .36246 Mdot
.41492 .36281 Mdot
.4151 .36278 Mdot
.41527 .36211 Mdot
.41544 .36195 Mdot
.41561 .36162 Mdot
.41577 .36126 Mdot
.41593 .36115 Mdot
.41611 .36133 Mdot
.41629 .36135 Mdot
.41648 .36155 Mdot
.41667 .3617 Mdot
.41684 .36106 Mdot
.41701 .36082 Mdot
.41718 .36054 Mdot
.41734 .36018 Mdot
.4175 .36 Mdot
.41768 .36017 Mdot
.41786 .36021 Mdot
.41804 .36033 Mdot
.41824 .36069 Mdot
.41842 .36008 Mdot
.41859 .35976 Mdot
.41875 .35956 Mdot
.41891 .35918 Mdot
.41908 .35895 Mdot
.41925 .35911 Mdot
.41943 .35917 Mdot
.41961 .35925 Mdot
.41982 .35983 Mdot
.41999 .35926 Mdot
.42017 .35886 Mdot
.42033 .3587 Mdot
.42049 .35831 Mdot
.42066 .35805 Mdot
.42083 .35819 Mdot
.42101 .35827 Mdot
.42119 .35833 Mdot
.4214 .35892 Mdot
.42158 .3584 Mdot
.42175 .35795 Mdot
.42191 .35782 Mdot
.42207 .35743 Mdot
.42224 .35715 Mdot
.42241 .35727 Mdot
.42259 .35737 Mdot
.42277 .3574 Mdot
.42297 .35801 Mdot
.42315 .35757 Mdot
.42333 .35705 Mdot
.42349 .35696 Mdot
.42365 .35657 Mdot
.42382 .35626 Mdot
.42399 .35633 Mdot
.42417 .35647 Mdot
.42435 .35649 Mdot
.42455 .35708 Mdot
.42473 .35675 Mdot
.4249 .35617 Mdot
.42507 .35608 Mdot
.42523 .3557 Mdot
.42539 .35537 Mdot
.42556 .35539 Mdot
.42574 .35554 Mdot
.42592 .35556 Mdot
.42612 .35609 Mdot
.4263 .35588 Mdot
.42647 .35524 Mdot
.42664 .35514 Mdot
.4268 .35477 Mdot
.42696 .35442 Mdot
.42713 .35439 Mdot
.42731 .35455 Mdot
.42749 .35457 Mdot
.42768 .35496 Mdot
.42787 .3549 Mdot
.42804 .35422 Mdot
.42821 .35408 Mdot
.42838 .35374 Mdot
.42853 .35338 Mdot
.4287 .35329 Mdot
.42888 .35346 Mdot
.42906 .35349 Mdot
.42925 .35371 Mdot
.42944 .35382 Mdot
.42961 .35317 Mdot
.42978 .35295 Mdot
.42995 .35266 Mdot
.43011 .3523 Mdot
.43027 .35214 Mdot
.43045 .35231 Mdot
.43062 .35235 Mdot
.43081 .35248 Mdot
.43101 .35279 Mdot
.43118 .35217 Mdot
.43135 .35187 Mdot
.43152 .35165 Mdot
.43168 .35128 Mdot
.43184 .35106 Mdot
.43202 .35122 Mdot
.4322 .35128 Mdot
.43238 .35136 Mdot
.43258 .35192 Mdot
.43276 .35135 Mdot
.43293 .35096 Mdot
.4331 .35079 Mdot
.43326 .3504 Mdot
.43342 .35015 Mdot
.4336 .3503 Mdot
.43378 .35037 Mdot
.43396 .35043 Mdot
.43416 .35103 Mdot
.43434 .35048 Mdot
.43451 .35004 Mdot
.43468 .34991 Mdot
.43484 .34952 Mdot
.435 .34924 Mdot
.43518 .34937 Mdot
.43536 .34946 Mdot
.43554 .34951 Mdot
.43574 .35011 Mdot
.43592 .34964 Mdot
.43609 .34915 Mdot
.43626 .34904 Mdot
.43642 .34866 Mdot
.43658 .34835 Mdot
.43676 .34844 Mdot
.43694 .34857 Mdot
.43712 .34859 Mdot
.43732 .3492 Mdot
.4375 .34883 Mdot
.43767 .34827 Mdot
.43784 .34818 Mdot
.438 .3478 Mdot
.43816 .34747 Mdot
.43833 .34751 Mdot
.43851 .34766 Mdot
.43869 .34767 Mdot
.43889 .34822 Mdot
.43907 .34798 Mdot
.43924 .34736 Mdot
.43941 .34726 Mdot
.43957 .34689 Mdot
.43973 .34654 Mdot
.4399 .34653 Mdot
.44008 .34669 Mdot
.44026 .3467 Mdot
.44046 .34716 Mdot
.44064 .34706 Mdot
.44081 .34638 Mdot
.44098 .34625 Mdot
.44115 .3459 Mdot
.4413 .34554 Mdot
.44147 .34547 Mdot
.44165 .34564 Mdot
.44183 .34566 Mdot
.44202 .34593 Mdot
.44221 .34598 Mdot
.44238 .34532 Mdot
.44255 .34512 Mdot
.44272 .34482 Mdot
.44287 .34446 Mdot
.44304 .34432 Mdot
.44322 .34449 Mdot
.44339 .34452 Mdot
.44358 .34468 Mdot
.44377 .34492 Mdot
.44395 .34429 Mdot
.44412 .34402 Mdot
.44429 .34378 Mdot
.44445 .34341 Mdot
.44461 .34321 Mdot
.44479 .34337 Mdot
.44497 .34342 Mdot
.44515 .34352 Mdot
.44535 .34399 Mdot
.44553 .34341 Mdot
.4457 .34304 Mdot
.44587 .34287 Mdot
.44602 .34248 Mdot
.44619 .34223 Mdot
.44637 .34239 Mdot
.44654 .34246 Mdot
.44673 .34252 Mdot
.44693 .34311 Mdot
.44711 .34255 Mdot
.44728 .34213 Mdot
.44745 .34199 Mdot
.44761 .3416 Mdot
.44777 .34132 Mdot
.44795 .34146 Mdot
.44812 .34154 Mdot
.44831 .34159 Mdot
.44851 .34219 Mdot
.44869 .3417 Mdot
.44886 .34122 Mdot
.44903 .34112 Mdot
.44919 .34073 Mdot
.44935 .34043 Mdot
.44952 .34053 Mdot
.4497 .34065 Mdot
.44988 .34068 Mdot
.45008 .34128 Mdot
.45026 .34089 Mdot
.45043 .34035 Mdot
.4506 .34026 Mdot
.45076 .33987 Mdot
.45092 .33955 Mdot
.4511 .3396 Mdot
.45128 .33974 Mdot
.45145 .33976 Mdot
.45165 .34032 Mdot
.45184 .34005 Mdot
.45201 .33944 Mdot
.45218 .33935 Mdot
.45234 .33897 Mdot
.4525 .33863 Mdot
.45267 .33863 Mdot
.45285 .33879 Mdot
.45302 .3388 Mdot
.45322 .3393 Mdot
.45341 .33916 Mdot
.45358 .33849 Mdot
.45375 .33836 Mdot
.45391 .33801 Mdot
.45407 .33765 Mdot
.45424 .33759 Mdot
.45442 .33776 Mdot
.45459 .33778 Mdot
.45479 .33809 Mdot
.45497 .3381 Mdot
.45515 .33743 Mdot
.45532 .33726 Mdot
.45548 .33694 Mdot
.45564 .33658 Mdot
.45581 .33645 Mdot
.45599 .33663 Mdot
.45616 .33666 Mdot
.45635 .33683 Mdot
.45654 .33703 Mdot
.45672 .3364 Mdot
.45689 .33614 Mdot
.45706 .33588 Mdot
.45721 .33551 Mdot
.45738 .33532 Mdot
.45756 .33549 Mdot
.45773 .33553 Mdot
.45792 .33564 Mdot
.45811 .33606 Mdot
.45829 .33546 Mdot
.45846 .33512 Mdot
.45863 .33493 Mdot
.45879 .33455 Mdot
.45895 .33431 Mdot
.45913 .33447 Mdot
.45931 .33453 Mdot
.45949 .3346 Mdot
.45969 .33519 Mdot
.45987 .33461 Mdot
.46004 .3342 Mdot
.46021 .33405 Mdot
.46037 .33366 Mdot
.46053 .33339 Mdot
.46071 .33354 Mdot
.46089 .33362 Mdot
.46107 .33367 Mdot
.46127 .33427 Mdot
.46145 .33376 Mdot
.46162 .33329 Mdot
.46179 .33318 Mdot
.46195 .3328 Mdot
.46211 .3325 Mdot
.46229 .33261 Mdot
.46247 .33272 Mdot
.46265 .33275 Mdot
.46285 .33335 Mdot
.46303 .33294 Mdot
.4632 .33241 Mdot
.46337 .33232 Mdot
.46353 .33193 Mdot
.46369 .33161 Mdot
.46386 .33168 Mdot
.46404 .33181 Mdot
.46422 .33183 Mdot
.46442 .33241 Mdot
.4646 .33211 Mdot
.46478 .33151 Mdot
.46494 .33142 Mdot
.46511 .33105 Mdot
.46527 .33071 Mdot
.46544 .33072 Mdot
.46562 .33087 Mdot
.46579 .33089 Mdot
.46599 .3314 Mdot
.46617 .33123 Mdot
.46635 .33057 Mdot
.46651 .33046 Mdot
.46668 .3301 Mdot
.46684 .32974 Mdot
.46701 .3297 Mdot
.46719 .32987 Mdot
.46736 .32988 Mdot
.46756 .33022 Mdot
.46774 .3302 Mdot
.46792 .32953 Mdot
.46808 .32937 Mdot
.46825 .32904 Mdot
.46841 .32868 Mdot
.46858 .32857 Mdot
.46875 .32875 Mdot
.46893 .32877 Mdot
.46912 .32897 Mdot
.46931 .32913 Mdot
.46948 .32848 Mdot
.46965 .32824 Mdot
.46982 .32796 Mdot
.46998 .3276 Mdot
.47015 .32742 Mdot
.47032 .32759 Mdot
.4705 .32763 Mdot
.47068 .32775 Mdot
.47088 .32812 Mdot
.47106 .32751 Mdot
.47123 .32719 Mdot
.4714 .32699 Mdot
.47155 .32661 Mdot
.47172 .32638 Mdot
.4719 .32654 Mdot
.47207 .3266 Mdot
.47226 .32667 Mdot
.47246 .32726 Mdot
.47264 .32668 Mdot
.47281 .32628 Mdot
.47298 .32613 Mdot
.47314 .32573 Mdot
.4733 .32547 Mdot
.47348 .32562 Mdot
.47365 .3257 Mdot
.47384 .32575 Mdot
.47404 .32635 Mdot
.47422 .32582 Mdot
.47439 .32537 Mdot
.47456 .32525 Mdot
.47472 .32486 Mdot
.47488 .32457 Mdot
.47505 .32469 Mdot
.47523 .32479 Mdot
.47541 .32483 Mdot
.47562 .32543 Mdot
.4758 .325 Mdot
.47597 .32448 Mdot
.47613 .32438 Mdot
.4763 .324 Mdot
.47646 .32369 Mdot
.47663 .32376 Mdot
.47681 .32389 Mdot
.47699 .32391 Mdot
.47719 .3245 Mdot
.47737 .32418 Mdot
.47754 .32359 Mdot
.47771 .3235 Mdot
.47787 .32313 Mdot
.47803 .32279 Mdot
.4782 .32281 Mdot
.47838 .32296 Mdot
.47856 .32298 Mdot
.47876 .3235 Mdot
.47894 .32331 Mdot
.47911 .32266 Mdot
.47928 .32256 Mdot
.47945 .32219 Mdot
.47961 .32184 Mdot
.47977 .32181 Mdot
.47995 .32197 Mdot
.48013 .32199 Mdot
.48033 .32237 Mdot
.48051 .32232 Mdot
.48068 .32164 Mdot
.48085 .32149 Mdot
.48102 .32116 Mdot
.48118 .3208 Mdot
.48134 .3207 Mdot
.48152 .32088 Mdot
.4817 .3209 Mdot
.48189 .32112 Mdot
.48208 .32124 Mdot
.48225 .32059 Mdot
.48242 .32036 Mdot
.48259 .32008 Mdot
.48275 .31972 Mdot
.48291 .31955 Mdot
.48309 .31972 Mdot
.48327 .31976 Mdot
.48345 .31989 Mdot
.48365 .32021 Mdot
.48382 .31959 Mdot
.48399 .31928 Mdot
.48416 .31907 Mdot
.48432 .3187 Mdot
.48449 .31848 Mdot
.48466 .31864 Mdot
.48484 .3187 Mdot
.48502 .31878 Mdot
.48522 .31935 Mdot
.4854 .31878 Mdot
.48557 .31839 Mdot
.48574 .31822 Mdot
.4859 .31783 Mdot
.48607 .31758 Mdot
.48624 .31773 Mdot
.48642 .3178 Mdot
.4866 .31786 Mdot
.4868 .31846 Mdot
.48698 .31791 Mdot
.48715 .31747 Mdot
.48732 .31734 Mdot
.48748 .31695 Mdot
.48765 .31667 Mdot
.48782 .3168 Mdot
.488 .31689 Mdot
.48818 .31694 Mdot
.48838 .31754 Mdot
.48856 .31707 Mdot
.48873 .31658 Mdot
.4889 .31647 Mdot
.48906 .31609 Mdot
.48922 .31578 Mdot
.4894 .31587 Mdot
.48958 .316 Mdot
.48976 .31602 Mdot
.48996 .31662 Mdot
.49014 .31627 Mdot
.49031 .3157 Mdot
.49048 .31561 Mdot
.49064 .31523 Mdot
.4908 .3149 Mdot
.49097 .31494 Mdot
.49115 .31509 Mdot
.49133 .3151 Mdot
.49153 .31565 Mdot
.49171 .31541 Mdot
.49188 .31478 Mdot
.49205 .31468 Mdot
.49221 .31432 Mdot
.49237 .31397 Mdot
.49254 .31395 Mdot
.49272 .31411 Mdot
.4929 .31413 Mdot
.4931 .31458 Mdot
.49328 .31448 Mdot
.49345 .3138 Mdot
.49362 .31367 Mdot
.49379 .31332 Mdot
.49395 .31296 Mdot
.49411 .31289 Mdot
.49429 .31306 Mdot
.49447 .31308 Mdot
.49466 .31334 Mdot
.49485 .3134 Mdot
.49502 .31274 Mdot
.49519 .31254 Mdot
.49536 .31224 Mdot
.49552 .31188 Mdot
.49568 .31174 Mdot
.49586 .31191 Mdot
.49604 .31194 Mdot
.49622 .3121 Mdot
.49642 .31235 Mdot
.49659 .31172 Mdot
.49676 .31144 Mdot
.49693 .3112 Mdot
.49709 .31083 Mdot
.49725 .31063 Mdot
.49743 .3108 Mdot
.49761 .31085 Mdot
.49779 .31094 Mdot
.49799 .31142 Mdot
.49817 .31084 Mdot
.49834 .31047 Mdot
.49851 .3103 Mdot
.49867 .30991 Mdot
.49883 .30966 Mdot
.49901 .30982 Mdot
.49919 .30989 Mdot
.49937 .30995 Mdot
.49957 .31054 Mdot
.49975 .30998 Mdot
.49992 .30956 Mdot
.50009 .30942 Mdot
.50025 .30903 Mdot
.50041 .30875 Mdot
.50059 .30889 Mdot
.50077 .30898 Mdot
.50095 .30902 Mdot
.50115 .30962 Mdot
.50133 .30914 Mdot
.5015 .30866 Mdot
.50167 .30855 Mdot
.50183 .30817 Mdot
.50199 .30786 Mdot
.50216 .30796 Mdot
.50234 .30808 Mdot
.50252 .30811 Mdot
.50272 .30871 Mdot
.5029 .30833 Mdot
.50308 .30778 Mdot
.50324 .30769 Mdot
.50341 .30731 Mdot
.50357 .30698 Mdot
.50374 .30703 Mdot
.50392 .30717 Mdot
.5041 .30719 Mdot
.5043 .30775 Mdot
.50448 .30749 Mdot
.50465 .30687 Mdot
.50482 .30678 Mdot
.50498 .3064 Mdot
.50514 .30606 Mdot
.50531 .30606 Mdot
.50549 .30622 Mdot
.50567 .30623 Mdot
.50587 .30672 Mdot
.50605 .30659 Mdot
.50622 .30591 Mdot
.50639 .30579 Mdot
.50655 .30544 Mdot
.50671 .30508 Mdot
.50688 .30502 Mdot
.50706 .30519 Mdot
.50724 .30521 Mdot
.50743 .30551 Mdot
.50762 .30553 Mdot
.50779 .30486 Mdot
.50796 .30468 Mdot
.50813 .30437 Mdot
.50828 .30401 Mdot
.50845 .30388 Mdot
.50863 .30405 Mdot
.5088 .30408 Mdot
.50899 .30426 Mdot
.50918 .30446 Mdot
.50936 .30383 Mdot
.50953 .30356 Mdot
.5097 .30331 Mdot
.50986 .30294 Mdot
.51002 .30275 Mdot
.5102 .30292 Mdot
.51037 .30296 Mdot
.51056 .30307 Mdot
.51076 .30349 Mdot
.51093 .3029 Mdot
.5111 .30255 Mdot
.51127 .30236 Mdot
.51143 .30198 Mdot
.5116 .30174 Mdot
.51177 .3019 Mdot
.51195 .30196 Mdot
.51213 .30203 Mdot
.51233 .30262 Mdot
.51251 .30204 Mdot
.51268 .30163 Mdot
.51285 .30149 Mdot
.51301 .30109 Mdot
.51318 .30082 Mdot
.51335 .30097 Mdot
.51353 .30105 Mdot
.51371 .3011 Mdot
.51391 .3017 Mdot
.51409 .30119 Mdot
.51427 .30073 Mdot
.51443 .30061 Mdot
.51459 .30023 Mdot
.51476 .29993 Mdot
.51493 .30004 Mdot
.51511 .30015 Mdot
.51529 .30018 Mdot
.51549 .30078 Mdot
.51567 .30038 Mdot
.51584 .29984 Mdot
.51601 .29975 Mdot
.51617 .29937 Mdot
.51633 .29905 Mdot
.51651 .29911 Mdot
.51669 .29925 Mdot
.51686 .29927 Mdot
.51706 .29984 Mdot
.51724 .29955 Mdot
.51742 .29895 Mdot
.51758 .29885 Mdot
.51775 .29848 Mdot
.51791 .29814 Mdot
.51808 .29815 Mdot
.51826 .2983 Mdot
.51843 .29832 Mdot
.51863 .29883 Mdot
.51882 .29866 Mdot
.51899 .298 Mdot
.51916 .29789 Mdot
.51932 .29753 Mdot
.51948 .29717 Mdot
.51965 .29712 Mdot
.51983 .29729 Mdot
.52 .29731 Mdot
.5202 .29764 Mdot
.52038 .29763 Mdot
.52056 .29696 Mdot
.52073 .29679 Mdot
.52089 .29647 Mdot
.52105 .29611 Mdot
.52122 .296 Mdot
.52139 .29617 Mdot
.52157 .2962 Mdot
.52176 .29639 Mdot
.52195 .29655 Mdot
.52213 .29591 Mdot
.52229 .29567 Mdot
.52246 .29539 Mdot
.52262 .29503 Mdot
.52279 .29485 Mdot
.52296 .29502 Mdot
.52314 .29506 Mdot
.52333 .29518 Mdot
.52352 .29555 Mdot
.5237 .29494 Mdot
.52387 .29462 Mdot
.52404 .29442 Mdot
.5242 .29404 Mdot
.52436 .29381 Mdot
.52454 .29397 Mdot
.52472 .29403 Mdot
.5249 .2941 Mdot
.5251 .29469 Mdot
.52528 .29411 Mdot
.52545 .29371 Mdot
.52562 .29356 Mdot
.52578 .29316 Mdot
.52594 .2929 Mdot
.52612 .29305 Mdot
.5263 .29312 Mdot
.52648 .29318 Mdot
.52668 .29378 Mdot
.52686 .29325 Mdot
.52703 .2928 Mdot
.5272 .29268 Mdot
.52736 .29229 Mdot
.52752 .292 Mdot
.5277 .29212 Mdot
.52788 .29222 Mdot
.52806 .29226 Mdot
.52826 .29286 Mdot
.52844 .29243 Mdot
.52861 .29191 Mdot
.52878 .29181 Mdot
.52894 .29143 Mdot
.5291 .29112 Mdot
.52927 .29119 Mdot
.52945 .29132 Mdot
.52963 .29134 Mdot
.52983 .29193 Mdot
.53001 .29161 Mdot
.53018 .29102 Mdot
.53035 .29093 Mdot
.53051 .29055 Mdot
.53067 .29022 Mdot
.53085 .29024 Mdot
.53102 .29039 Mdot
.5312 .29041 Mdot
.5314 .29093 Mdot
.53158 .29074 Mdot
.53176 .29009 Mdot
.53192 .28998 Mdot
.53209 .28962 Mdot
.53225 .28927 Mdot
.53242 .28923 Mdot
.53259 .2894 Mdot
.53277 .28941 Mdot
.53297 .28979 Mdot
.53315 .28974 Mdot
.53333 .28907 Mdot
.53349 .28891 Mdot
.53366 .28858 Mdot
.53382 .28822 Mdot
.53399 .28812 Mdot
.53416 .2883 Mdot
.53434 .28832 Mdot
.53453 .28854 Mdot
.53472 .28866 Mdot
.53489 .28801 Mdot
.53506 .28779 Mdot
.53523 .2875 Mdot
.53539 .28714 Mdot
.53555 .28697 Mdot
.53573 .28714 Mdot
.53591 .28718 Mdot
.53609 .28731 Mdot
.53629 .28763 Mdot
.53647 .28702 Mdot
.53663 .28671 Mdot
.5368 .2865 Mdot
.53696 .28613 Mdot
.53713 .2859 Mdot
.5373 .28607 Mdot
.53748 .28612 Mdot
.53766 .2862 Mdot
.53787 .28678 Mdot
.53804 .28621 Mdot
.53822 .28582 Mdot
.53838 .28565 Mdot
.53854 .28526 Mdot
.53871 .285 Mdot
.53888 .28515 Mdot
.53906 .28523 Mdot
.53924 .28529 Mdot
.53945 .28588 Mdot
.53963 .28534 Mdot
.5398 .2849 Mdot
.53996 .28477 Mdot
.54012 .28438 Mdot
.54029 .2841 Mdot
.54046 .28423 Mdot
.54064 .28432 Mdot
.54082 .28436 Mdot
.54102 .28496 Mdot
.5412 .2845 Mdot
.54138 .28401 Mdot
.54154 .2839 Mdot
.5417 .28352 Mdot
.54187 .28321 Mdot
.54204 .2833 Mdot
.54222 .28342 Mdot
.5424 .28345 Mdot
.5426 .28405 Mdot
.54278 .28369 Mdot
.54295 .28313 Mdot
.54312 .28304 Mdot
.54328 .28266 Mdot
.54344 .28233 Mdot
.54361 .28236 Mdot
.54379 .28251 Mdot
.54397 .28253 Mdot
.54417 .28307 Mdot
.54435 .28284 Mdot
.54453 .28221 Mdot
.54469 .28211 Mdot
.54486 .28174 Mdot
.54502 .28139 Mdot
.54519 .28137 Mdot
.54536 .28154 Mdot
.54554 .28155 Mdot
.54574 .282 Mdot
.54592 .2819 Mdot
.5461 .28122 Mdot
.54626 .28109 Mdot
.54643 .28074 Mdot
.54659 .28038 Mdot
.54676 .2803 Mdot
.54693 .28048 Mdot
.54711 .2805 Mdot
.5473 .28076 Mdot
.54749 .28082 Mdot
.54766 .28016 Mdot
.54783 .27996 Mdot
.548 .27966 Mdot
.54816 .2793 Mdot
.54832 .27915 Mdot
.5485 .27933 Mdot
.54868 .27936 Mdot
.54886 .27951 Mdot
.54906 .27977 Mdot
.54923 .27914 Mdot
.5494 .27886 Mdot
.54957 .27863 Mdot
.54973 .27826 Mdot
.5499 .27805 Mdot
.55007 .27822 Mdot
.55025 .27827 Mdot
.55043 .27836 Mdot
.55063 .27885 Mdot
.55081 .27827 Mdot
.55098 .2779 Mdot
.55115 .27773 Mdot
.55131 .27734 Mdot
.55147 .27709 Mdot
.55165 .27725 Mdot
.55183 .27731 Mdot
.55201 .27738 Mdot
.55221 .27797 Mdot
.55239 .27741 Mdot
.55256 .27699 Mdot
.55273 .27685 Mdot
.55289 .27646 Mdot
.55305 .27618 Mdot
.55323 .27632 Mdot
.55341 .2764 Mdot
.55359 .27645 Mdot
.55379 .27705 Mdot
.55397 .27657 Mdot
.55414 .27609 Mdot
.55431 .27598 Mdot
.55447 .2756 Mdot
.55463 .27529 Mdot
.55481 .27539 Mdot
.55499 .27551 Mdot
.55517 .27554 Mdot
.55537 .27614 Mdot
.55555 .27576 Mdot
.55572 .27521 Mdot
.55589 .27512 Mdot
.55605 .27474 Mdot
.55621 .27441 Mdot
.55638 .27446 Mdot
.55656 .2746 Mdot
.55674 .27462 Mdot
.55694 .27518 Mdot
.55712 .27492 Mdot
.55729 .2743 Mdot
.55746 .27421 Mdot
.55762 .27383 Mdot
.55778 .27349 Mdot
.55795 .27348 Mdot
.55813 .27364 Mdot
.55831 .27366 Mdot
.55851 .27415 Mdot
.55869 .27402 Mdot
.55886 .27334 Mdot
.55903 .27322 Mdot
.5592 .27287 Mdot
.55936 .27251 Mdot
.55952 .27245 Mdot
.5597 .27262 Mdot
.55988 .27263 Mdot
.56007 .27293 Mdot
.56026 .27295 Mdot
.56043 .27229 Mdot
.5606 .27211 Mdot
.56077 .2718 Mdot
.56093 .27143 Mdot
.56109 .2713 Mdot
.56127 .27148 Mdot
.56145 .27151 Mdot
.56163 .27168 Mdot
.56183 .27189 Mdot
.562 .27125 Mdot
.56217 .27099 Mdot
.56234 .27074 Mdot
.5625 .27037 Mdot
.56266 .27017 Mdot
.56284 .27034 Mdot
.56302 .27039 Mdot
.5632 .27049 Mdot
.5634 .27092 Mdot
.56358 .27033 Mdot
.56375 .26998 Mdot
.56392 .2698 Mdot
.56407 .26941 Mdot
.56424 .26917 Mdot
.56441 .26933 Mdot
.56459 .26939 Mdot
.56478 .26946 Mdot
.56498 .27005 Mdot
.56516 .26948 Mdot
.56533 .26906 Mdot
.5655 .26892 Mdot
.56566 .26853 Mdot
.56582 .26826 Mdot
.566 .2684 Mdot
.56617 .26848 Mdot
.56636 .26853 Mdot
.56656 .26913 Mdot
.56674 .26863 Mdot
.56691 .26816 Mdot
.56708 .26804 Mdot
.56724 .26766 Mdot
.5674 .26736 Mdot
.56757 .26747 Mdot
.56775 .26758 Mdot
.56793 .26761 Mdot
.56813 .26821 Mdot
.56831 .26781 Mdot
.56849 .26727 Mdot
.56865 .26718 Mdot
.56881 .2668 Mdot
.56898 .26648 Mdot
.56915 .26654 Mdot
.56933 .26668 Mdot
.56951 .2667 Mdot
.56971 .26726 Mdot
.56989 .26698 Mdot
.57006 .26638 Mdot
.57023 .26628 Mdot
.57039 .26591 Mdot
.57055 .26557 Mdot
.57072 .26557 Mdot
.5709 .26573 Mdot
.57108 .26575 Mdot
.57128 .26625 Mdot
.57146 .26609 Mdot
.57163 .26543 Mdot
.5718 .26531 Mdot
.57196 .26496 Mdot
.57212 .2646 Mdot
.57229 .26455 Mdot
.57247 .26472 Mdot
.57265 .26474 Mdot
.57284 .26507 Mdot
.57303 .26506 Mdot
.5732 .26439 Mdot
.57337 .26422 Mdot
.57353 .2639 Mdot
.57369 .26354 Mdot
.57386 .26342 Mdot
.57404 .2636 Mdot
.57421 .26362 Mdot
.5744 .26381 Mdot
.57459 .26398 Mdot
.57477 .26334 Mdot
.57494 .26309 Mdot
.57511 .26282 Mdot
.57526 .26246 Mdot
.57543 .26227 Mdot
.57561 .26245 Mdot
.57578 .26249 Mdot
.57597 .2626 Mdot
.57617 .26298 Mdot
.57634 .26238 Mdot
.57651 .26204 Mdot
.57668 .26185 Mdot
.57684 .26147 Mdot
.577 .26124 Mdot
.57718 .2614 Mdot
.57736 .26146 Mdot
.57754 .26153 Mdot
.57774 .26212 Mdot
.57792 .26154 Mdot
.57809 .26114 Mdot
.57826 .26098 Mdot
.57842 .26059 Mdot
.57858 .26033 Mdot
.57876 .26047 Mdot
.57894 .26055 Mdot
.57912 .26061 Mdot
.57932 .2612 Mdot
.5795 .26068 Mdot
.57967 .26023 Mdot
.57984 .26011 Mdot
.58 .25972 Mdot
.58016 .25943 Mdot
.58034 .25954 Mdot
.58052 .25965 Mdot
.5807 .25968 Mdot
.5809 .26029 Mdot
.58108 .25986 Mdot
.58125 .25934 Mdot
.58142 .25924 Mdot
.58158 .25886 Mdot
.58174 .25855 Mdot
.58191 .25861 Mdot
.5821 .25875 Mdot
.58227 .25877 Mdot
.58247 .25935 Mdot
.58265 .25904 Mdot
.58283 .25845 Mdot
.58299 .25836 Mdot
.58316 .25798 Mdot
.58332 .25765 Mdot
.58349 .25767 Mdot
.58367 .25782 Mdot
.58384 .25783 Mdot
.58404 .25835 Mdot
.58422 .25817 Mdot
.5844 .25752 Mdot
.58457 .25741 Mdot
.58473 .25704 Mdot
.58489 .25669 Mdot
.58506 .25666 Mdot
.58524 .25682 Mdot
.58541 .25684 Mdot
.58561 .25721 Mdot
.58579 .25716 Mdot
.58597 .25649 Mdot
.58614 .25634 Mdot
.5863 .25601 Mdot
.58646 .25565 Mdot
.58663 .25555 Mdot
.5868 .25572 Mdot
.58698 .25574 Mdot
.58717 .25596 Mdot
.58736 .25608 Mdot
.58754 .25544 Mdot
.5877 .25521 Mdot
.58787 .25492 Mdot
.58803 .25456 Mdot
.5882 .25439 Mdot
.58837 .25457 Mdot
.58855 .2546 Mdot
.58874 .25473 Mdot
.58893 .25506 Mdot
.58911 .25445 Mdot
.58928 .25414 Mdot
.58945 .25393 Mdot
.5896 .25355 Mdot
.58977 .25333 Mdot
.58995 .25349 Mdot
.59012 .25355 Mdot
.59031 .25363 Mdot
.59051 .25421 Mdot
.59069 .25364 Mdot
.59086 .25324 Mdot
.59103 .25308 Mdot
.59119 .25268 Mdot
.59135 .25243 Mdot
.59153 .25258 Mdot
.5917 .25265 Mdot
.59189 .25271 Mdot
.59209 .25331 Mdot
.59227 .25277 Mdot
.59244 .25233 Mdot
.59261 .2522 Mdot
.59277 .25181 Mdot
.59293 .25152 Mdot
.59311 .25165 Mdot
.59329 .25175 Mdot
.59347 .25179 Mdot
.59367 .25239 Mdot
.59385 .25193 Mdot
.59402 .25143 Mdot
.59419 .25133 Mdot
.59435 .25095 Mdot
.59451 .25064 Mdot
.59468 .25072 Mdot
.59486 .25085 Mdot
.59504 .25087 Mdot
.59524 .25147 Mdot
.59542 .25112 Mdot
.59559 .25055 Mdot
.59576 .25046 Mdot
.59592 .25008 Mdot
.59608 .24975 Mdot
.59626 .24978 Mdot
.59644 .24993 Mdot
.59661 .24995 Mdot
.59681 .25049 Mdot
.59699 .25026 Mdot
.59717 .24963 Mdot
.59733 .24953 Mdot
.5975 .24916 Mdot
.59766 .24881 Mdot
.59783 .24879 Mdot
.59801 .24896 Mdot
.59818 .24897 Mdot
.59838 .2494 Mdot
.59856 .24931 Mdot
.59874 .24864 Mdot
.5989 .2485 Mdot
.59907 .24816 Mdot
.59923 .2478 Mdot
.5994 .24772 Mdot
.59957 .24789 Mdot
.59975 .24791 Mdot
.59994 .24816 Mdot
.60013 .24824 Mdot
.60031 .24758 Mdot
.60047 .24737 Mdot
.60064 .24708 Mdot
.6008 .24671 Mdot
.60097 .24656 Mdot
.60114 .24674 Mdot
.60132 .24677 Mdot
.60151 .24692 Mdot
.6017 .24719 Mdot
.60188 .24656 Mdot
.60204 .24627 Mdot
.60221 .24605 Mdot
.60237 .24568 Mdot
.60254 .24547 Mdot
.60271 .24563 Mdot
.60289 .24568 Mdot
.60307 .24577 Mdot
.60327 .24628 Mdot
.60345 .2457 Mdot
.60362 .24533 Mdot
.60379 .24516 Mdot
.60395 .24477 Mdot
.60412 .24452 Mdot
.60429 .24467 Mdot
.60447 .24474 Mdot
.60465 .2448 Mdot
.60485 .2454 Mdot
.60503 .24484 Mdot
.6052 .24441 Mdot
.60537 .24428 Mdot
.60553 .24389 Mdot
.6057 .24361 Mdot
.60587 .24374 Mdot
.60605 .24383 Mdot
.60623 .24388 Mdot
.60643 .24448 Mdot
.60661 .244 Mdot
.60678 .24351 Mdot
.60695 .24341 Mdot
.60711 .24303 Mdot
.60727 .24272 Mdot
.60745 .24281 Mdot
.60763 .24294 Mdot
.60781 .24296 Mdot
.60801 .24357 Mdot
.60819 .24319 Mdot
.60836 .24264 Mdot
.60853 .24255 Mdot
.60869 .24217 Mdot
.60885 .24184 Mdot
.60902 .24188 Mdot
.6092 .24203 Mdot
.60938 .24205 Mdot
.60958 .2426 Mdot
.60976 .24235 Mdot
.60993 .24173 Mdot
.6101 .24163 Mdot
.61026 .24126 Mdot
.61042 .24092 Mdot
.61059 .24091 Mdot
.61077 .24107 Mdot
.61095 .24108 Mdot
.61115 .24157 Mdot
.61133 .24145 Mdot
.61151 .24077 Mdot
.61167 .24064 Mdot
.61184 .24029 Mdot
.612 .23993 Mdot
.61217 .23987 Mdot
.61234 .24004 Mdot
.61252 .24006 Mdot
.61271 .24035 Mdot
.6129 .24038 Mdot
.61307 .23971 Mdot
.61324 .23953 Mdot
.61341 .23922 Mdot
.61357 .23886 Mdot
.61373 .23872 Mdot
.61391 .2389 Mdot
.61409 .23893 Mdot
.61428 .2391 Mdot
.61447 .23931 Mdot
.61464 .23868 Mdot
.61481 .23841 Mdot
.61498 .23816 Mdot
.61514 .2378 Mdot
.6153 .2376 Mdot
.61548 .23777 Mdot
.61566 .23781 Mdot
.61584 .23791 Mdot
.61604 .23835 Mdot
.61622 .23776 Mdot
.61639 .23741 Mdot
.61656 .23723 Mdot
.61672 .23684 Mdot
.61688 .2366 Mdot
.61706 .23676 Mdot
.61723 .23682 Mdot
.61742 .23689 Mdot
.61762 .23748 Mdot
.6178 .23691 Mdot
.61797 .23649 Mdot
.61814 .23635 Mdot
.6183 .23596 Mdot
.61846 .23568 Mdot
.61864 .23582 Mdot
.61882 .23591 Mdot
.619 .23596 Mdot
.6192 .23656 Mdot
.61938 .23606 Mdot
.61955 .23559 Mdot
.61972 .23547 Mdot
.61988 .23509 Mdot
.62004 .23479 Mdot
.62021 .23489 Mdot
.6204 .23501 Mdot
.62057 .23504 Mdot
.62078 .23564 Mdot
.62096 .23524 Mdot
.62113 .23471 Mdot
.62129 .23461 Mdot
.62146 .23423 Mdot
.62162 .23391 Mdot
.62179 .23397 Mdot
.62197 .23411 Mdot
.62215 .23412 Mdot
.62235 .23469 Mdot
.62253 .23441 Mdot
.6227 .23381 Mdot
.62287 .23371 Mdot
.62303 .23334 Mdot
.62319 .233 Mdot
.62336 .233 Mdot
.62354 .23316 Mdot
.62372 .23317 Mdot
.62392 .23368 Mdot
.6241 .23352 Mdot
.62427 .23286 Mdot
.62444 .23274 Mdot
.62461 .23238 Mdot
.62476 .23203 Mdot
.62493 .23198 Mdot
.62511 .23215 Mdot
.62529 .23216 Mdot
.62548 .23249 Mdot
.62567 .23248 Mdot
.62584 .23181 Mdot
.62601 .23164 Mdot
.62618 .23132 Mdot
.62633 .23096 Mdot
.6265 .23084 Mdot
.62668 .23102 Mdot
.62685 .23104 Mdot
.62704 .23123 Mdot
.62723 .23141 Mdot
.62741 .23077 Mdot
.62758 .23052 Mdot
.62775 .23025 Mdot
.62791 .22989 Mdot
.62807 .2297 Mdot
.62825 .22987 Mdot
.62842 .22991 Mdot
.62861 .23003 Mdot
.62881 .23041 Mdot
.62898 .22981 Mdot
.62915 .22948 Mdot
.62932 .22928 Mdot
.62948 .2289 Mdot
.62965 .22867 Mdot
.62982 .22883 Mdot
.63 .22889 Mdot
.63018 .22896 Mdot
.63038 .22955 Mdot
.63056 .22897 Mdot
.63073 .22857 Mdot
.6309 .22841 Mdot
.63106 .22802 Mdot
.63123 .22776 Mdot
.6314 .2279 Mdot
.63158 .22798 Mdot
.63176 .22803 Mdot
.63196 .22863 Mdot
.63214 .22812 Mdot
.63231 .22766 Mdot
.63248 .22754 Mdot
.63264 .22715 Mdot
.63281 .22686 Mdot
.63298 .22697 Mdot
.63316 .22708 Mdot
.63334 .22711 Mdot
.63354 .22772 Mdot
.63372 .22729 Mdot
.63389 .22677 Mdot
.63406 .22667 Mdot
.63422 .22629 Mdot
.63438 .22598 Mdot
.63456 .22604 Mdot
.63474 .22618 Mdot
.63491 .2262 Mdot
.63511 .22678 Mdot
.6353 .22647 Mdot
.63547 .22588 Mdot
.63564 .22579 Mdot
.6358 .22541 Mdot
.63596 .22507 Mdot
.63613 .22509 Mdot
.63631 .22524 Mdot
.63649 .22526 Mdot
.63668 .22578 Mdot
.63687 .2256 Mdot
.63704 .22494 Mdot
.63721 .22483 Mdot
.63737 .22447 Mdot
.63753 .22412 Mdot
.6377 .22408 Mdot
.63788 .22425 Mdot
.63805 .22426 Mdot
.63825 .22463 Mdot
.63843 .22459 Mdot
.63861 .22391 Mdot
.63878 .22376 Mdot
.63894 .22343 Mdot
.6391 .22307 Mdot
.63927 .22296 Mdot
.63945 .22314 Mdot
.63962 .22316 Mdot
.63981 .22337 Mdot
.64 .22351 Mdot
.64018 .22286 Mdot
.64035 .22263 Mdot
.64051 .22235 Mdot
.64067 .22199 Mdot
.64084 .22181 Mdot
.64101 .22199 Mdot
.64119 .22202 Mdot
.64138 .22215 Mdot
.64157 .22249 Mdot
.64175 .22188 Mdot
.64192 .22156 Mdot
.64209 .22136 Mdot
.64225 .22098 Mdot
.64241 .22075 Mdot
.64259 .22092 Mdot
.64276 .22097 Mdot
.64295 .22105 Mdot
.64315 .22163 Mdot
.64333 .22106 Mdot
.6435 .22066 Mdot
.64367 .2205 Mdot
.64383 .22011 Mdot
.64399 .21985 Mdot
.64417 .22 Mdot
.64435 .22008 Mdot
.64453 .22013 Mdot
.64473 .22073 Mdot
.64491 .22019 Mdot
.64508 .21975 Mdot
.64525 .21962 Mdot
.64541 .21924 Mdot
.64557 .21895 Mdot
.64575 .21907 Mdot
.64593 .21917 Mdot
.64611 .21921 Mdot
.64631 .21981 Mdot
.64649 .21936 Mdot
.64666 .21886 Mdot
.64683 .21876 Mdot
.64699 .21837 Mdot
.64715 .21806 Mdot
.64732 .21814 Mdot
.6475 .21827 Mdot
.64768 .2183 Mdot
.64788 .21889 Mdot
.64806 .21855 Mdot
.64824 .21798 Mdot
.6484 .21789 Mdot
.64856 .21751 Mdot
.64873 .21717 Mdot
.6489 .2172 Mdot
.64908 .21735 Mdot
.64925 .21737 Mdot
.64945 .2179 Mdot
.64963 .21769 Mdot
.64981 .21705 Mdot
.64998 .21695 Mdot
.65014 .21658 Mdot
.6503 .21623 Mdot
.65047 .21621 Mdot
.65065 .21637 Mdot
.65082 .21639 Mdot
.65102 .21681 Mdot
.6512 .21673 Mdot
.65138 .21605 Mdot
.65155 .21591 Mdot
.65171 .21557 Mdot
.65187 .21521 Mdot
.65204 .21512 Mdot
.65222 .2153 Mdot
.65239 .21532 Mdot
.65258 .21556 Mdot
.65277 .21565 Mdot
.65295 .21499 Mdot
.65311 .21478 Mdot
.65328 .21449 Mdot
.65344 .21413 Mdot
.65361 .21397 Mdot
.65378 .21414 Mdot
.65396 .21418 Mdot
.65415 .21432 Mdot
.65434 .2146 Mdot
.65452 .21398 Mdot
.65469 .21369 Mdot
.65486 .21347 Mdot
.65501 .21309 Mdot
.65518 .21288 Mdot
.65535 .21304 Mdot
.65553 .2131 Mdot
.65572 .21318 Mdot
.65592 .21371 Mdot
.65609 .21313 Mdot
.65626 .21275 Mdot
.65643 .21259 Mdot
.65659 .21219 Mdot
.65676 .21194 Mdot
.65693 .2121 Mdot
.65711 .21217 Mdot
.65729 .21223 Mdot
.6575 .21282 Mdot
.65767 .21227 Mdot
.65785 .21184 Mdot
.65801 .21171 Mdot
.65817 .21132 Mdot
.65834 .21104 Mdot
.65851 .21117 Mdot
.65869 .21126 Mdot
.65887 .2113 Mdot
.65907 .2119 Mdot
.65925 .21143 Mdot
.65943 .21094 Mdot
.65959 .21084 Mdot
.65975 .21046 Mdot
.65992 .21015 Mdot
.66009 .21024 Mdot
.66027 .21036 Mdot
.66045 .21039 Mdot
.66065 .211 Mdot
.66083 .21063 Mdot
.661 .21007 Mdot
.66117 .20998 Mdot
.66133 .2096 Mdot
.66149 .20927 Mdot
.66166 .20931 Mdot
.66184 .20946 Mdot
.66202 .20947 Mdot
.66222 .21003 Mdot
.6624 .20978 Mdot
.66258 .20916 Mdot
.66274 .20906 Mdot
.66291 .20869 Mdot
.66307 .20834 Mdot
.66324 .20833 Mdot
.66341 .20849 Mdot
.66359 .20851 Mdot
.66379 .20899 Mdot
.66397 .20887 Mdot
.66415 .20819 Mdot
.66431 .20806 Mdot
.66448 .20771 Mdot
.66464 .20735 Mdot
.66481 .20729 Mdot
.66499 .20746 Mdot
.66516 .20748 Mdot
.66535 .20776 Mdot
.66554 .2078 Mdot
.66572 .20713 Mdot
.66588 .20694 Mdot
.66605 .20664 Mdot
.66621 .20628 Mdot
.66638 .20614 Mdot
.66655 .20631 Mdot
.66673 .20634 Mdot
.66692 .20651 Mdot
.66711 .20673 Mdot
.66729 .2061 Mdot
.66745 .20583 Mdot
.66762 .20559 Mdot
.66778 .20522 Mdot
.66795 .20502 Mdot
.66812 .20519 Mdot
.6683 .20523 Mdot
.66848 .20533 Mdot
.66868 .20578 Mdot
.66886 .20519 Mdot
.66903 .20484 Mdot
.6692 .20466 Mdot
.66936 .20427 Mdot
.66952 .20403 Mdot
.6697 .20419 Mdot
.66988 .20425 Mdot
.67006 .20432 Mdot
.67026 .20491 Mdot
.67044 .20434 Mdot
.67061 .20392 Mdot
.67078 .20378 Mdot
.67094 .20339 Mdot
.6711 .20312 Mdot
.67128 .20325 Mdot
.67146 .20334 Mdot
.67164 .20339 Mdot
.67184 .20399 Mdot
.67202 .20349 Mdot
.67219 .20302 Mdot
.67236 .20291 Mdot
.67252 .20252 Mdot
.67268 .20222 Mdot
.67286 .20232 Mdot
.67304 .20244 Mdot
.67322 .20247 Mdot
.67342 .20307 Mdot
.6736 .20268 Mdot
.67377 .20214 Mdot
.67394 .20205 Mdot
.6741 .20166 Mdot
.67426 .20134 Mdot
.67443 .2014 Mdot
.67461 .20154 Mdot
.67479 .20155 Mdot
.67499 .20212 Mdot
.67517 .20184 Mdot
.67534 .20124 Mdot
.67551 .20114 Mdot
.67567 .20077 Mdot
.67583 .20043 Mdot
.676 .20043 Mdot
.67618 .20059 Mdot
.67636 .2006 Mdot
.67656 .2011 Mdot
.67674 .20095 Mdot
.67691 .20028 Mdot
.67708 .20017 Mdot
.67725 .19981 Mdot
.67741 .19945 Mdot
.67757 .1994 Mdot
.67775 .19957 Mdot
.67793 .19959 Mdot
.67812 .1999 Mdot
.67831 .19991 Mdot
.67848 .19924 Mdot
.67865 .19906 Mdot
.67882 .19875 Mdot
.67898 .19839 Mdot
.67914 .19826 Mdot
.67932 .19844 Mdot
.6795 .19847 Mdot
.67969 .19865 Mdot
.67988 .19883 Mdot
.68005 .19819 Mdot
.68022 .19794 Mdot
.68039 .19768 Mdot
.68055 .19731 Mdot
.68071 .19712 Mdot
.68089 .19729 Mdot
.68107 .19734 Mdot
.68125 .19745 Mdot
.68145 .19784 Mdot
.68163 .19725 Mdot
.6818 .19691 Mdot
.68196 .19672 Mdot
.68212 .19633 Mdot
.68229 .1961 Mdot
.68246 .19626 Mdot
.68264 .19632 Mdot
.68283 .19639 Mdot
.68303 .19698 Mdot
.68321 .1964 Mdot
.68338 .196 Mdot
.68355 .19585 Mdot
.6837 .19545 Mdot
.68387 .19519 Mdot
.68404 .19533 Mdot
.68422 .19541 Mdot
.68441 .19546 Mdot
.68461 .19606 Mdot
.68479 .19555 Mdot
.68496 .19509 Mdot
.68513 .19497 Mdot
.68528 .19459 Mdot
.68545 .19429 Mdot
.68562 .1944 Mdot
.6858 .19451 Mdot
.68598 .19454 Mdot
.68618 .19515 Mdot
.68636 .19473 Mdot
.68653 .1942 Mdot
.6867 .19411 Mdot
.68686 .19372 Mdot
.68703 .19341 Mdot
.6872 .19347 Mdot
.68738 .19361 Mdot
.68756 .19363 Mdot
.68776 .19421 Mdot
.68794 .1939 Mdot
.68811 .19331 Mdot
.68828 .19322 Mdot
.68844 .19284 Mdot
.6886 .1925 Mdot
.68877 .19252 Mdot
.68895 .19267 Mdot
.68913 .19269 Mdot
.68933 .1932 Mdot
.68951 .19303 Mdot
.68968 .19237 Mdot
.68985 .19226 Mdot
.69001 .1919 Mdot
.69017 .19154 Mdot
.69034 .1915 Mdot
.69052 .19167 Mdot
.6907 .19169 Mdot
.69089 .19204 Mdot
.69108 .19201 Mdot
.69125 .19134 Mdot
.69142 .19118 Mdot
.69158 .19085 Mdot
.69174 .19049 Mdot
.69191 .19038 Mdot
.69209 .19056 Mdot
.69226 .19058 Mdot
.69245 .19079 Mdot
.69264 .19093 Mdot
.69282 .19028 Mdot
.69299 .19005 Mdot
.69316 .18977 Mdot
.69331 .18941 Mdot
.69348 .18923 Mdot
.69366 .1894 Mdot
.69383 .18944 Mdot
.69402 .18956 Mdot
.69421 .18991 Mdot
.69439 .18931 Mdot
.69456 .18899 Mdot
.69473 .18878 Mdot
.69489 .18841 Mdot
.69505 .18818 Mdot
.69523 .18834 Mdot
.69541 .1884 Mdot
.69559 .18848 Mdot
.69579 .18906 Mdot
.69597 .18848 Mdot
.69614 .18809 Mdot
.69631 .18793 Mdot
.69647 .18753 Mdot
.69663 .18727 Mdot
.69681 .18742 Mdot
.69699 .1875 Mdot
.69717 .18755 Mdot
.69737 .18815 Mdot
.69755 .18762 Mdot
.69772 .18717 Mdot
.69789 .18705 Mdot
.69805 .18666 Mdot
.69821 .18637 Mdot
.69839 .18649 Mdot
.69857 .18659 Mdot
.69875 .18663 Mdot
.69895 .18723 Mdot
.69913 .18679 Mdot
.6993 .18628 Mdot
.69947 .18618 Mdot
.69963 .1858 Mdot
.69979 .18549 Mdot
.69996 .18556 Mdot
.70015 .1857 Mdot
.70032 .18572 Mdot
.70052 .18631 Mdot
.7007 .18598 Mdot
.70088 .1854 Mdot
.70104 .18531 Mdot
.70121 .18493 Mdot
.70137 .1846 Mdot
.70154 .18462 Mdot
.70172 .18477 Mdot
.70189 .18479 Mdot
.70209 .18532 Mdot
.70228 .18511 Mdot
.70245 .18447 Mdot
.70262 .18437 Mdot
.70278 .184 Mdot
.70294 .18365 Mdot
.70311 .18362 Mdot
.70329 .18379 Mdot
.70346 .1838 Mdot
.70366 .1842 Mdot
.70384 .18414 Mdot
.70402 .18346 Mdot
.70419 .18331 Mdot
.70435 .18298 Mdot
.70451 .18262 Mdot
.70468 .18253 Mdot
.70486 .1827 Mdot
.70503 .18272 Mdot
.70522 .18296 Mdot
.70541 .18306 Mdot
.70559 .1824 Mdot
.70575 .18219 Mdot
.70592 .1819 Mdot
.70608 .18153 Mdot
.70625 .18137 Mdot
.70642 .18155 Mdot
.7066 .18158 Mdot
.70679 .18172 Mdot
.70698 .18202 Mdot
.70716 .1814 Mdot
.70733 .1811 Mdot
.7075 .18088 Mdot
.70765 .18051 Mdot
.70782 .18029 Mdot
.708 .18046 Mdot
.70817 .18051 Mdot
.70836 .18059 Mdot
.70856 .18114 Mdot
.70874 .18057 Mdot
.70891 .18018 Mdot
.70908 .18002 Mdot
.70923 .17962 Mdot
.7094 .17937 Mdot
.70958 .17952 Mdot
.70975 .1796 Mdot
.70994 .17966 Mdot
.71014 .18025 Mdot
.71032 .1797 Mdot
.71049 .17927 Mdot
.71066 .17914 Mdot
.71082 .17875 Mdot
.71098 .17846 Mdot
.71116 .1786 Mdot
.71133 .17869 Mdot
.71152 .17873 Mdot
.71172 .17933 Mdot
.7119 .17886 Mdot
.71207 .17837 Mdot
.71224 .17827 Mdot
.7124 .17789 Mdot
.71256 .17758 Mdot
.71273 .17766 Mdot
.71291 .17779 Mdot
.71309 .17782 Mdot
.71329 .17842 Mdot
.71347 .17806 Mdot
.71364 .1775 Mdot
.71381 .17741 Mdot
.71397 .17702 Mdot
.71413 .1767 Mdot
.71431 .17674 Mdot
.71449 .17688 Mdot
.71466 .1769 Mdot
.71486 .17745 Mdot
.71504 .17721 Mdot
.71522 .17658 Mdot
.71538 .17648 Mdot
.71555 .17611 Mdot
.71571 .17577 Mdot
.71588 .17575 Mdot
.71606 .17592 Mdot
.71623 .17593 Mdot
.71643 .1764 Mdot
.71661 .17629 Mdot
.71679 .17561 Mdot
.71696 .17548 Mdot
.71712 .17513 Mdot
.71728 .17477 Mdot
.71745 .1747 Mdot
.71763 .17487 Mdot
.7178 .17489 Mdot
.718 .17517 Mdot
.71818 .17521 Mdot
.71836 .17455 Mdot
.71852 .17436 Mdot
.71869 .17405 Mdot
.71885 .17369 Mdot
.71902 .17355 Mdot
.71919 .17373 Mdot
.71937 .17376 Mdot
.71956 .17392 Mdot
.71975 .17415 Mdot
.71993 .17352 Mdot
.72009 .17325 Mdot
.72026 .17301 Mdot
.72042 .17264 Mdot
.72059 .17244 Mdot
.72076 .1726 Mdot
.72094 .17265 Mdot
.72113 .17275 Mdot
.72132 .17321 Mdot
.7215 .17263 Mdot
.72167 .17227 Mdot
.72184 .17209 Mdot
.722 .1717 Mdot
.72216 .17146 Mdot
.72234 .17161 Mdot
.72252 .17168 Mdot
.7227 .17175 Mdot
.7229 .17234 Mdot
.72308 .17177 Mdot
.72325 .17135 Mdot
.72342 .17121 Mdot
.72358 .17082 Mdot
.72374 .17055 Mdot
.72392 .17068 Mdot
.7241 .17077 Mdot
.72428 .17082 Mdot
.72448 .17142 Mdot
.72466 .17093 Mdot
.72483 .17045 Mdot
.725 .17034 Mdot
.72516 .16996 Mdot
.72532 .16965 Mdot
.7255 .16975 Mdot
.72568 .16987 Mdot
.72586 .1699 Mdot
.72606 .1705 Mdot
.72624 .17011 Mdot
.72641 .16957 Mdot
.72658 .16948 Mdot
.72674 .1691 Mdot
.7269 .16877 Mdot
.72707 .16882 Mdot
.72725 .16897 Mdot
.72743 .16898 Mdot
.72763 .16955 Mdot
.72781 .16928 Mdot
.72798 .16866 Mdot
.72815 .16857 Mdot
.72831 .1682 Mdot
.72847 .16786 Mdot
.72864 .16785 Mdot
.72882 .16801 Mdot
.729 .16803 Mdot
.7292 .16852 Mdot
.72938 .16838 Mdot
.72956 .16771 Mdot
.72972 .16759 Mdot
.72989 .16724 Mdot
.73005 .16688 Mdot
.73022 .16682 Mdot
.73039 .16699 Mdot
.73057 .16701 Mdot
.73076 .16732 Mdot
.73095 .16733 Mdot
.73112 .16666 Mdot
.73129 .16648 Mdot
.73146 .16617 Mdot
.73162 .16581 Mdot
.73178 .16568 Mdot
.73196 .16586 Mdot
.73214 .16589 Mdot
.73233 .16606 Mdot
.73252 .16626 Mdot
.73269 .16562 Mdot
.73286 .16536 Mdot
.73303 .1651 Mdot
.73319 .16474 Mdot
.73335 .16455 Mdot
.73353 .16472 Mdot
.73371 .16476 Mdot
.73389 .16487 Mdot
.73409 .16528 Mdot
.73427 .16468 Mdot
.73444 .16434 Mdot
.73461 .16415 Mdot
.73477 .16377 Mdot
.73493 .16353 Mdot
.73511 .16369 Mdot
.73528 .16375 Mdot
.73547 .16382 Mdot
.73567 .16441 Mdot
.73585 .16384 Mdot
.73602 .16343 Mdot
.73619 .16328 Mdot
.73635 .16289 Mdot
.73651 .16262 Mdot
.73669 .16276 Mdot
.73686 .16284 Mdot
.73705 .16289 Mdot
.73725 .16349 Mdot
.73743 .16298 Mdot
.7376 .16252 Mdot
.73777 .1624 Mdot
.73793 .16202 Mdot
.73809 .16172 Mdot
.73826 .16183 Mdot
.73844 .16194 Mdot
.73862 .16197 Mdot
.73882 .16258 Mdot
.739 .16216 Mdot
.73918 .16163 Mdot
.73934 .16154 Mdot
.7395 .16116 Mdot
.73967 .16084 Mdot
.73984 .1609 Mdot
.74002 .16104 Mdot
.7402 .16106 Mdot
.7404 .16163 Mdot
.74058 .16134 Mdot
.74075 .16074 Mdot
.74092 .16065 Mdot
.74108 .16027 Mdot
.74124 .15993 Mdot
.74141 .15994 Mdot
.74159 .1601 Mdot
.74177 .16011 Mdot
.74197 .16062 Mdot
.74215 .16046 Mdot
.74232 .1598 Mdot
.74249 .15968 Mdot
.74265 .15932 Mdot
.74281 .15897 Mdot
.74298 .15892 Mdot
.74316 .15909 Mdot
.74334 .15911 Mdot
.74353 .15945 Mdot
.74372 .15943 Mdot
.74389 .15876 Mdot
.74406 .15859 Mdot
.74423 .15827 Mdot
.74438 .15791 Mdot
.74455 .1578 Mdot
.74473 .15797 Mdot
.7449 .158 Mdot
.74509 .1582 Mdot
.74528 .15835 Mdot
.74546 .15771 Mdot
.74563 .15747 Mdot
.7458 .15719 Mdot
.74596 .15683 Mdot
.74612 .15665 Mdot
.7463 .15682 Mdot
.74647 .15686 Mdot
.74666 .15698 Mdot
.74686 .15734 Mdot
.74703 .15674 Mdot
.7472 .15641 Mdot
.74737 .15621 Mdot
.74753 .15583 Mdot
.7477 .1556 Mdot
.74787 .15577 Mdot
.74805 .15582 Mdot
.74823 .1559 Mdot
.74843 .15648 Mdot
.74861 .15591 Mdot
.74878 .15551 Mdot
.74895 .15535 Mdot
.74911 .15496 Mdot
.74928 .1547 Mdot
.74945 .15485 Mdot
.74963 .15492 Mdot
.74981 .15498 Mdot
.75001 .15557 Mdot
.75019 .15505 Mdot
.75036 .1546 Mdot
.75053 .15447 Mdot
.75069 .15409 Mdot
.75085 .1538 Mdot
.75103 .15392 Mdot
.75121 .15402 Mdot
.75139 .15405 Mdot
.75159 .15466 Mdot
.75177 .15422 Mdot
.75194 .15371 Mdot
.75211 .15361 Mdot
.75227 .15323 Mdot
.75243 .15291 Mdot
.75261 .15298 Mdot
.75279 .15312 Mdot
.75296 .15314 Mdot
.75317 .15373 Mdot
.75335 .1534 Mdot
.75352 .15282 Mdot
.75369 .15273 Mdot
.75385 .15235 Mdot
.75401 .15202 Mdot
.75418 .15204 Mdot
.75436 .15219 Mdot
.75454 .15221 Mdot
.75474 .15273 Mdot
.75492 .15254 Mdot
.75509 .15189 Mdot
.75526 .15178 Mdot
.75542 .15142 Mdot
.75558 .15107 Mdot
.75575 .15104 Mdot
.75593 .1512 Mdot
.7561 .15122 Mdot
.7563 .15161 Mdot
.75649 .15155 Mdot
.75666 .15087 Mdot
.75683 .15072 Mdot
.75699 .15039 Mdot
.75715 .15003 Mdot
.75732 .14993 Mdot
.7575 .15011 Mdot
.75767 .15013 Mdot
.75786 .15036 Mdot
.75805 .15047 Mdot
.75823 .14982 Mdot
.7584 .14959 Mdot
.75856 .14931 Mdot
.75872 .14895 Mdot
.75889 .14878 Mdot
.75907 .14895 Mdot
.75924 .14899 Mdot
.75943 .14912 Mdot
.75962 .14943 Mdot
.7598 .14882 Mdot
.75997 .14851 Mdot
.76014 .1483 Mdot
.7603 .14793 Mdot
.76046 .14771 Mdot
.76064 .14787 Mdot
.76081 .14793 Mdot
.761 .14801 Mdot
.7612 .14857 Mdot
.76138 .148 Mdot
.76155 .14761 Mdot
.76172 .14745 Mdot
.76188 .14705 Mdot
.76204 .1468 Mdot
.76222 .14695 Mdot
.7624 .14702 Mdot
.76258 .14708 Mdot
.76278 .14768 Mdot
.76296 .14713 Mdot
.76313 .1467 Mdot
.7633 .14657 Mdot
.76346 .14618 Mdot
.76362 .14589 Mdot
.7638 .14602 Mdot
.76398 .14612 Mdot
.76416 .14616 Mdot
.76436 .14676 Mdot
.76454 .1463 Mdot
.76471 .1458 Mdot
.76488 .1457 Mdot
.76504 .14532 Mdot
.7652 .14501 Mdot
.76537 .14509 Mdot
.76555 .14522 Mdot
.76573 .14525 Mdot
.76593 .14585 Mdot
.76611 .14549 Mdot
.76629 .14493 Mdot
.76645 .14483 Mdot
.76662 .14445 Mdot
.76678 .14412 Mdot
.76695 .14416 Mdot
.76713 .14431 Mdot
.7673 .14433 Mdot
.7675 .14487 Mdot
.76769 .14464 Mdot
.76786 .14401 Mdot
.76803 .14391 Mdot
.76819 .14354 Mdot
.76835 .14319 Mdot
.76852 .14318 Mdot
.7687 .14334 Mdot
.76887 .14335 Mdot
.76907 .14381 Mdot
.76926 .14371 Mdot
.76943 .14303 Mdot
.7696 .1429 Mdot
.76976 .14255 Mdot
.76992 .14219 Mdot
.77009 .14212 Mdot
.77027 .14229 Mdot
.77044 .14231 Mdot
.77064 .14258 Mdot
.77082 .14263 Mdot
.771 .14197 Mdot
.77117 .14177 Mdot
.77133 .14147 Mdot
.77149 .14111 Mdot
.77166 .14097 Mdot
.77184 .14114 Mdot
.77201 .14117 Mdot
.7722 .14133 Mdot
.77239 .14157 Mdot
.77257 .14095 Mdot
.77274 .14067 Mdot
.77291 .14043 Mdot
.77306 .14006 Mdot
.77323 .13986 Mdot
.77341 .14002 Mdot
.77358 .14007 Mdot
.77377 .14017 Mdot
.77397 .14065 Mdot
.77414 .14006 Mdot
.77431 .1397 Mdot
.77448 .13952 Mdot
.77464 .13913 Mdot
.77481 .13889 Mdot
.77498 .13904 Mdot
.77516 .13911 Mdot
.77534 .13917 Mdot
.77554 .13977 Mdot
.77572 .1392 Mdot
.77589 .13878 Mdot
.77606 .13864 Mdot
.77622 .13825 Mdot
.77639 .13798 Mdot
.77656 .13811 Mdot
.77674 .1382 Mdot
.77692 .13824 Mdot
.77712 .13885 Mdot
.7773 .13836 Mdot
.77747 .13788 Mdot
.77764 .13777 Mdot
.7778 .13739 Mdot
.77797 .13709 Mdot
.77814 .13718 Mdot
.77832 .1373 Mdot
.7785 .13733 Mdot
.7787 .13793 Mdot
.77888 .13755 Mdot
.77905 .137 Mdot
.77922 .13691 Mdot
.77938 .13653 Mdot
.77954 .1362 Mdot
.77971 .13625 Mdot
.77989 .1364 Mdot
.78007 .13641 Mdot
.78027 .13697 Mdot
.78045 .13671 Mdot
.78063 .1361 Mdot
.78079 .136 Mdot
.78096 .13563 Mdot
.78112 .13528 Mdot
.78129 .13528 Mdot
.78146 .13544 Mdot
.78164 .13545 Mdot
.78184 .13595 Mdot
.78202 .13582 Mdot
.7822 .13514 Mdot
.78237 .13502 Mdot
.78253 .13466 Mdot
.78269 .13431 Mdot
.78286 .13425 Mdot
.78304 .13442 Mdot
.78321 .13443 Mdot
.78341 .13474 Mdot
.78359 .13475 Mdot
.78377 .13409 Mdot
.78393 .13391 Mdot
.7841 .13359 Mdot
.78426 .13323 Mdot
.78443 .13311 Mdot
.7846 .13328 Mdot
.78478 .13331 Mdot
.78497 .13348 Mdot
.78516 .13368 Mdot
.78533 .13305 Mdot
.7855 .13279 Mdot
.78567 .13253 Mdot
.78583 .13217 Mdot
.786 .13197 Mdot
.78617 .13214 Mdot
.78635 .13219 Mdot
.78653 .13229 Mdot
.78673 .13271 Mdot
.78691 .13212 Mdot
.78708 .13177 Mdot
.78725 .13159 Mdot
.78741 .1312 Mdot
.78757 .13096 Mdot
.78775 .13112 Mdot
.78793 .13119 Mdot
.78811 .13125 Mdot
.78831 .13184 Mdot
.78849 .13127 Mdot
.78866 .13086 Mdot
.78883 .13071 Mdot
.78899 .13032 Mdot
.78915 .13005 Mdot
.78933 .13019 Mdot
.78951 .13027 Mdot
.78969 .13032 Mdot
.78989 .13092 Mdot
.79007 .13042 Mdot
.79024 .12995 Mdot
.79041 .12983 Mdot
.79057 .12945 Mdot
.79073 .12915 Mdot
.79091 .12926 Mdot
.79109 .12937 Mdot
.79127 .1294 Mdot
.79147 .13001 Mdot
.79165 .1296 Mdot
.79182 .12907 Mdot
.79199 .12897 Mdot
.79215 .12859 Mdot
.79231 .12827 Mdot
.79248 .12833 Mdot
.79266 .12847 Mdot
.79284 .12849 Mdot
.79304 .12906 Mdot
.79322 .12877 Mdot
.79339 .12817 Mdot
.79356 .12808 Mdot
.79372 .1277 Mdot
.79388 .12736 Mdot
.79405 .12737 Mdot
.79423 .12753 Mdot
.79441 .12754 Mdot
.79461 .12805 Mdot
.79479 .12789 Mdot
.79497 .12722 Mdot
.79513 .12711 Mdot
.7953 .12675 Mdot
.79546 .1264 Mdot
.79562 .12635 Mdot
.7958 .12652 Mdot
.79598 .12653 Mdot
.79617 .12687 Mdot
.79636 .12685 Mdot
.79653 .12618 Mdot
.7967 .12602 Mdot
.79687 .1257 Mdot
.79703 .12534 Mdot
.79719 .12522 Mdot
.79737 .1254 Mdot
.79755 .12542 Mdot
.79774 .12562 Mdot
.79793 .12578 Mdot
.7981 .12514 Mdot
.79827 .12489 Mdot
.79844 .12462 Mdot
.7986 .12426 Mdot
.79876 .12407 Mdot
.79894 .12425 Mdot
.79912 .12429 Mdot
.7993 .1244 Mdot
.7995 .12477 Mdot
.79968 .12417 Mdot
.79984 .12384 Mdot
.80001 .12364 Mdot
.80017 .12326 Mdot
.80034 .12303 Mdot
.80051 .12319 Mdot
.80069 .12325 Mdot
.80087 .12333 Mdot
.80108 .12391 Mdot
.80125 .12334 Mdot
.80143 .12293 Mdot
.80159 .12278 Mdot
.80175 .12239 Mdot
.80192 .12212 Mdot
.80209 .12227 Mdot
.80227 .12235 Mdot
.80245 .1224 Mdot
.80266 .123 Mdot
.80283 .12248 Mdot
.80301 .12202 Mdot
.80317 .1219 Mdot
.80333 .12152 Mdot
.8035 .12123 Mdot
.80367 .12134 Mdot
.80385 .12144 Mdot
.80403 .12148 Mdot
.80423 .12208 Mdot
.80441 .12165 Mdot
.80458 .12113 Mdot
.80475 .12104 Mdot
.80491 .12066 Mdot
.80507 .12034 Mdot
.80525 .12041 Mdot
.80543 .12055 Mdot
.80561 .12057 Mdot
.80581 .12115 Mdot
.80599 .12083 Mdot
.80616 .12025 Mdot
.80633 .12016 Mdot
.80649 .11978 Mdot
.80665 .11944 Mdot
.80682 .11946 Mdot
.807 .11962 Mdot
.80718 .11963 Mdot
.80738 .12016 Mdot
.80756 .11996 Mdot
.80773 .11931 Mdot
.8079 .11921 Mdot
.80806 .11884 Mdot
.80822 .11849 Mdot
.80839 .11846 Mdot
.80857 .11862 Mdot
.80875 .11864 Mdot
.80894 .11902 Mdot
.80913 .11897 Mdot
.8093 .11829 Mdot
.80947 .11814 Mdot
.80964 .11781 Mdot
.80979 .11745 Mdot
.80996 .11735 Mdot
.81014 .11753 Mdot
.81031 .11755 Mdot
.81051 .11777 Mdot
.81069 .11789 Mdot
.81087 .11724 Mdot
.81104 .11701 Mdot
.81121 .11673 Mdot
.81136 .11637 Mdot
.81153 .1162 Mdot
.81171 .11637 Mdot
.81188 .11641 Mdot
.81207 .11654 Mdot
.81226 .11686 Mdot
.81244 .11625 Mdot
.81261 .11594 Mdot
.81278 .11573 Mdot
.81294 .11535 Mdot
.8131 .11513 Mdot
.81328 .11529 Mdot
.81346 .11535 Mdot
.81364 .11543 Mdot
.81384 .11601 Mdot
.81402 .11544 Mdot
.81419 .11504 Mdot
.81436 .11488 Mdot
.81452 .11448 Mdot
.81468 .11423 Mdot
.81486 .11438 Mdot
.81504 .11445 Mdot
.81522 .11451 Mdot
.81542 .11511 Mdot
.8156 .11457 Mdot
.81577 .11413 Mdot
.81594 .114 Mdot
.8161 .11361 Mdot
.81626 .11332 Mdot
.81644 .11345 Mdot
.81662 .11355 Mdot
.8168 .11359 Mdot
.817 .11419 Mdot
.81718 .11373 Mdot
.81735 .11323 Mdot
.81752 .11313 Mdot
.81768 .11275 Mdot
.81784 .11244 Mdot
.81802 .11252 Mdot
.8182 .11265 Mdot
.81838 .11268 Mdot
.81858 .11328 Mdot
.81876 .11292 Mdot
.81893 .11236 Mdot
.8191 .11226 Mdot
.81926 .11188 Mdot
.81942 .11155 Mdot
.81959 .11159 Mdot
.81977 .11174 Mdot
.81995 .11176 Mdot
.82015 .1123 Mdot
.82033 .11207 Mdot
.8205 .11144 Mdot
.82067 .11134 Mdot
.82083 .11097 Mdot
.82099 .11062 Mdot
.82116 .1106 Mdot
.82134 .11077 Mdot
.82152 .11078 Mdot
.82171 .11123 Mdot
.8219 .11113 Mdot
.82207 .11045 Mdot
.82224 .11032 Mdot
.8224 .10997 Mdot
.82256 .10961 Mdot
.82273 .10954 Mdot
.82291 .10971 Mdot
.82309 .10973 Mdot
.82328 .10999 Mdot
.82346 .11005 Mdot
.82364 .10939 Mdot
.82381 .10919 Mdot
.82398 .10889 Mdot
.82413 .10853 Mdot
.8243 .10839 Mdot
.82448 .10856 Mdot
.82465 .10859 Mdot
.82484 .10875 Mdot
.82503 .109 Mdot
.82521 .10837 Mdot
.82538 .10809 Mdot
.82555 .10786 Mdot
.82571 .10749 Mdot
.82587 .10728 Mdot
.82605 .10745 Mdot
.82622 .1075 Mdot
.82641 .10759 Mdot
.82661 .10808 Mdot
.82679 .10749 Mdot
.82696 .10713 Mdot
.82713 .10695 Mdot
.82728 .10656 Mdot
.82745 .10632 Mdot
.82763 .10647 Mdot
.8278 .10654 Mdot
.82799 .1066 Mdot
.82819 .1072 Mdot
.82837 .10664 Mdot
.82854 .10621 Mdot
.82871 .10607 Mdot
.82887 .10568 Mdot
.82903 .10541 Mdot
.8292 .10554 Mdot
.82938 .10563 Mdot
.82957 .10568 Mdot
.82977 .10628 Mdot
.82995 .10579 Mdot
.83012 .10531 Mdot
.83029 .1052 Mdot
.83045 .10482 Mdot
.83061 .10452 Mdot
.83078 .10461 Mdot
.83096 .10473 Mdot
.83114 .10476 Mdot
.83134 .10537 Mdot
.83152 .10498 Mdot
.83169 .10443 Mdot
.83186 .10434 Mdot
.83202 .10396 Mdot
.83218 .10364 Mdot
.83236 .10368 Mdot
.83254 .10383 Mdot
.83271 .10385 Mdot
.83291 .1044 Mdot
.83309 .10414 Mdot
.83327 .10353 Mdot
.83344 .10343 Mdot
.8336 .10306 Mdot
.83376 .10272 Mdot
.83393 .10271 Mdot
.83411 .10287 Mdot
.83428 .10288 Mdot
.83448 .10338 Mdot
.83467 .10325 Mdot
.83484 .10257 Mdot
.83501 .10245 Mdot
.83517 .10209 Mdot
.83533 .10174 Mdot
.8355 .10167 Mdot
.83568 .10184 Mdot
.83585 .10186 Mdot
.83605 .10216 Mdot
.83623 .10218 Mdot
.83641 .10151 Mdot
.83658 .10133 Mdot
.83674 .10102 Mdot
.8369 .10066 Mdot
.83707 .10053 Mdot
.83725 .10071 Mdot
.83742 .10074 Mdot
.83761 .10091 Mdot
.8378 .10111 Mdot
.83798 .10048 Mdot
.83815 .10022 Mdot
.83831 .09996 Mdot
.83847 .0996 Mdot
.83864 .0994 Mdot
.83881 .09957 Mdot
.83899 .09962 Mdot
.83918 .09972 Mdot
.83937 .10015 Mdot
.83955 .09955 Mdot
.83972 .0992 Mdot
.83989 .09902 Mdot
.84005 .09864 Mdot
.84021 .0984 Mdot
.84039 .09855 Mdot
.84057 .09862 Mdot
.84075 .09869 Mdot
.84095 .09927 Mdot
.84113 .0987 Mdot
.8413 .09829 Mdot
.84147 .09814 Mdot
.84163 .09775 Mdot
.84179 .09748 Mdot
.84197 .09762 Mdot
.84215 .0977 Mdot
.84233 .09775 Mdot
.84253 .09835 Mdot
.84271 .09785 Mdot
.84288 .09738 Mdot
.84305 .09727 Mdot
.84321 .09688 Mdot
.84337 .09658 Mdot
.84355 .09669 Mdot
.84373 .0968 Mdot
.84391 .09683 Mdot
.84411 .09744 Mdot
.84429 .09703 Mdot
.84446 .0965 Mdot
.84463 .0964 Mdot
.84479 .09602 Mdot
.84495 .0957 Mdot
.84512 .09576 Mdot
.8453 .0959 Mdot
.84548 .09592 Mdot
.84568 .09649 Mdot
.84586 .0962 Mdot
.84604 .0956 Mdot
.8462 .09551 Mdot
.84636 .09513 Mdot
.84653 .09479 Mdot
.8467 .0948 Mdot
.84687 .09496 Mdot
.84705 .09497 Mdot
.84725 .09548 Mdot
.84743 .09532 Mdot
.84761 .09465 Mdot
.84777 .09454 Mdot
.84794 .09418 Mdot
.8481 .09383 Mdot
.84827 .09378 Mdot
.84844 .09395 Mdot
.84862 .09396 Mdot
.84882 .0943 Mdot
.849 .09428 Mdot
.84918 .09361 Mdot
.84934 .09345 Mdot
.84951 .09313 Mdot
.84967 .09276 Mdot
.84984 .09265 Mdot
.85001 .09282 Mdot
.85019 .09285 Mdot
.85038 .09304 Mdot
.85057 .09321 Mdot
.85074 .09257 Mdot
.85091 .09232 Mdot
.85108 .09205 Mdot
.85124 .09168 Mdot
.85141 .0915 Mdot
.85158 .09167 Mdot
.85176 .09171 Mdot
.85194 .09183 Mdot
.85214 .0922 Mdot
.85232 .0916 Mdot
.85249 .09127 Mdot
.85266 .09107 Mdot
.85281 .09069 Mdot
.85298 .09046 Mdot
.85316 .09062 Mdot
.85333 .09068 Mdot
.85352 .09076 Mdot
.85372 .09134 Mdot
.8539 .09077 Mdot
.85407 .09036 Mdot
.85424 .09021 Mdot
.8544 .08982 Mdot
.85456 .08955 Mdot
.85474 .0897 Mdot
.85491 .08978 Mdot
.8551 .08983 Mdot
.8553 .09043 Mdot
.85548 .08991 Mdot
.85565 .08945 Mdot
.85582 .08933 Mdot
.85598 .08895 Mdot
.85614 .08865 Mdot
.85631 .08877 Mdot
.85649 .08887 Mdot
.85667 .08891 Mdot
.85688 .08951 Mdot
.85705 .08908 Mdot
.85723 .08856 Mdot
.85739 .08847 Mdot
.85756 .08809 Mdot
.85772 .08777 Mdot
.85789 .08784 Mdot
.85807 .08797 Mdot
.85825 .088 Mdot
.85845 .08858 Mdot
.85863 .08826 Mdot
.8588 .08768 Mdot
.85897 .08759 Mdot
.85913 .08721 Mdot
.85929 .08687 Mdot
.85946 .08689 Mdot
.85964 .08704 Mdot
.85982 .08706 Mdot
.86002 .08758 Mdot
.8602 .08739 Mdot
.86037 .08674 Mdot
.86054 .08664 Mdot
.86071 .08627 Mdot
.86086 .08592 Mdot
.86103 .08588 Mdot
.86121 .08605 Mdot
.86139 .08607 Mdot
.86158 .08644 Mdot
.86177 .08639 Mdot
.86194 .08572 Mdot
.86211 .08557 Mdot
.86228 .08524 Mdot
.86244 .08488 Mdot
.8626 .08478 Mdot
.86278 .08495 Mdot
.86296 .08497 Mdot
.86315 .08519 Mdot
.86334 .08531 Mdot
.86351 .08466 Mdot
.86368 .08444 Mdot
.86385 .08415 Mdot
.86401 .08379 Mdot
.86417 .08362 Mdot
.86435 .0838 Mdot
.86453 .08383 Mdot
.86471 .08396 Mdot
.86491 .08429 Mdot
.86508 .08367 Mdot
.86525 .08336 Mdot
.86542 .08315 Mdot
.86558 .08278 Mdot
.86575 .08256 Mdot
.86592 .08272 Mdot
.8661 .08277 Mdot
.86628 .08285 Mdot
.86648 .08343 Mdot
.86666 .08286 Mdot
.86683 .08247 Mdot
.867 .08231 Mdot
.86716 .08191 Mdot
.86733 .08166 Mdot
.8675 .08181 Mdot
.86768 .08188 Mdot
.86786 .08194 Mdot
.86806 .08253 Mdot
.86824 .08199 Mdot
.86841 .08155 Mdot
.86858 .08142 Mdot
.86874 .08104 Mdot
.86891 .08075 Mdot
.86908 .08088 Mdot
.86926 .08097 Mdot
.86944 .08101 Mdot
.86964 .08162 Mdot
.86982 .08116 Mdot
.86999 .08066 Mdot
.87016 .08056 Mdot
.87032 .08018 Mdot
.87048 .07987 Mdot
.87066 .07995 Mdot
.87084 .08008 Mdot
.87102 .0801 Mdot
.87122 .0807 Mdot
.8714 .08035 Mdot
.87157 .07978 Mdot
.87174 .07969 Mdot
.8719 .07931 Mdot
.87206 .07898 Mdot
.87223 .07902 Mdot
.87241 .07916 Mdot
.87259 .07918 Mdot
.87279 .07972 Mdot
.87297 .07949 Mdot
.87314 .07886 Mdot
.87331 .07876 Mdot
.87347 .07839 Mdot
.87363 .07804 Mdot
.8738 .07803 Mdot
.87398 .07819 Mdot
.87416 .0782 Mdot
.87436 .07864 Mdot
.87454 .07855 Mdot
.87471 .07787 Mdot
.87488 .07774 Mdot
.87505 .07739 Mdot
.87521 .07703 Mdot
.87537 .07695 Mdot
.87555 .07713 Mdot
.87573 .07715 Mdot
.87592 .0774 Mdot
.87611 .07747 Mdot
.87628 .07681 Mdot
.87645 .07661 Mdot
.87662 .07631 Mdot
.87678 .07595 Mdot
.87694 .0758 Mdot
.87712 .07598 Mdot
.87729 .07601 Mdot
.87748 .07616 Mdot
.87768 .07642 Mdot
.87785 .0758 Mdot
.87802 .07551 Mdot
.87819 .07528 Mdot
.87835 .07491 Mdot
.87851 .0747 Mdot
.87869 .07487 Mdot
.87887 .07492 Mdot
.87905 .07501 Mdot
.87925 .07551 Mdot
.87943 .07493 Mdot
.8796 .07455 Mdot
.87977 .07438 Mdot
.87993 .07399 Mdot
.88009 .07375 Mdot
.88027 .0739 Mdot
.88045 .07397 Mdot
.88063 .07403 Mdot
.88083 .07462 Mdot
.88101 .07407 Mdot
.88118 .07364 Mdot
.88135 .0735 Mdot
.88151 .07311 Mdot
.88167 .07283 Mdot
.88185 .07297 Mdot
.88203 .07306 Mdot
.88221 .0731 Mdot
.88241 .0737 Mdot
.88259 .07322 Mdot
.88276 .07274 Mdot
.88293 .07263 Mdot
.88309 .07225 Mdot
.88325 .07195 Mdot
.88342 .07204 Mdot
.8836 .07216 Mdot
.88378 .07219 Mdot
.88398 .07279 Mdot
.88416 .07242 Mdot
.88434 .07186 Mdot
.8845 .07177 Mdot
.88467 .07139 Mdot
.88483 .07107 Mdot
.885 .07111 Mdot
.88518 .07126 Mdot
.88536 .07127 Mdot
.88556 .07183 Mdot
.88574 .07157 Mdot
.88591 .07095 Mdot
.88608 .07086 Mdot
.88624 .07049 Mdot
.8864 .07014 Mdot
.88657 .07014 Mdot
.88675 .0703 Mdot
.88693 .07031 Mdot
.88713 .0708 Mdot
.88731 .07067 Mdot
.88748 .06999 Mdot
.88765 .06987 Mdot
.88781 .06952 Mdot
.88797 .06916 Mdot
.88814 .0691 Mdot
.88832 .06927 Mdot
.8885 .06929 Mdot
.88869 .06958 Mdot
.88888 .06961 Mdot
.88905 .06894 Mdot
.88922 .06876 Mdot
.88939 .06845 Mdot
.88954 .06809 Mdot
.88971 .06795 Mdot
.88989 .06813 Mdot
.89006 .06816 Mdot
.89025 .06833 Mdot
.89044 .06854 Mdot
.89062 .06791 Mdot
.89079 .06764 Mdot
.89096 .06739 Mdot
.89111 .06702 Mdot
.89128 .06683 Mdot
.89146 .067 Mdot
.89163 .06704 Mdot
.89182 .06714 Mdot
.89202 .06758 Mdot
.89219 .06698 Mdot
.89236 .06663 Mdot
.89253 .06645 Mdot
.89269 .06607 Mdot
.89286 .06583 Mdot
.89303 .06598 Mdot
.89321 .06605 Mdot
.89339 .06612 Mdot
.89359 .0667 Mdot
.89377 .06613 Mdot
.89394 .06572 Mdot
.89411 .06557 Mdot
.89427 .06518 Mdot
.89444 .06491 Mdot
.89461 .06505 Mdot
.89479 .06513 Mdot
.89497 .06518 Mdot
.89517 .06578 Mdot
.89535 .06528 Mdot
.89552 .06481 Mdot
.89569 .0647 Mdot
.89585 .06431 Mdot
.89602 .06402 Mdot
.89619 .06412 Mdot
.89637 .06423 Mdot
.89655 .06426 Mdot
.89675 .06487 Mdot
.89693 .06446 Mdot
.8971 .06393 Mdot
.89727 .06384 Mdot
.89743 .06345 Mdot
.89759 .06313 Mdot
.89777 .06319 Mdot
.89795 .06333 Mdot
.89812 .06335 Mdot
.89832 .06392 Mdot
.8985 .06363 Mdot
.89868 .06303 Mdot
.89884 .06294 Mdot
.89901 .06256 Mdot
.89917 .06222 Mdot
.89934 .06223 Mdot
.89952 .06238 Mdot
.89969 .0624 Mdot
.89989 .0629 Mdot
.90007 .06275 Mdot
.90025 .06208 Mdot
.90042 .06197 Mdot
.90058 .06161 Mdot
.90074 .06125 Mdot
.90091 .0612 Mdot
.90109 .06137 Mdot
.90126 .06139 Mdot
.90146 .06172 Mdot
.90164 .06171 Mdot
.90182 .06104 Mdot
.90198 .06087 Mdot
.90215 .06055 Mdot
.90231 .06019 Mdot
.90248 .06007 Mdot
.90265 .06025 Mdot
.90283 .06027 Mdot
.90302 .06046 Mdot
.90321 .06063 Mdot
.90339 .05999 Mdot
.90355 .05975 Mdot
.90372 .05947 Mdot
.90388 .05911 Mdot
.90405 .05893 Mdot
.90422 .0591 Mdot
.9044 .05914 Mdot
.90459 .05925 Mdot
.90478 .05963 Mdot
.90496 .05903 Mdot
.90513 .0587 Mdot
.9053 .0585 Mdot
.90546 .05812 Mdot
.90562 .05789 Mdot
.9058 .05805 Mdot
.90598 .05811 Mdot
.90616 .05819 Mdot
.90636 .05877 Mdot
.90654 .0582 Mdot
.90671 .05779 Mdot
.90688 .05764 Mdot
.90704 .05725 Mdot
.9072 .05698 Mdot
.90738 .05713 Mdot
.90756 .05721 Mdot
.90774 .05726 Mdot
.90794 .05786 Mdot
.90812 .05734 Mdot
.90829 .05688 Mdot
.90846 .05676 Mdot
.90862 .05638 Mdot
.90878 .05608 Mdot
.90896 .0562 Mdot
.90914 .0563 Mdot
.90932 .05634 Mdot
.90952 .05694 Mdot
.9097 .05651 Mdot
.90987 .05599 Mdot
.91004 .0559 Mdot
.9102 .05552 Mdot
.91036 .0552 Mdot
.91053 .05527 Mdot
.91071 .0554 Mdot
.91089 .05542 Mdot
.91109 .05601 Mdot
.91127 .05569 Mdot
.91144 .0551 Mdot
.91161 .05501 Mdot
.91177 .05464 Mdot
.91193 .0543 Mdot
.9121 .05432 Mdot
.91228 .05447 Mdot
.91246 .05449 Mdot
.91266 .05501 Mdot
.91284 .05482 Mdot
.91302 .05417 Mdot
.91318 .05406 Mdot
.91335 .0537 Mdot
.91351 .05335 Mdot
.91368 .05331 Mdot
.91385 .05348 Mdot
.91403 .05349 Mdot
.91423 .05386 Mdot
.91441 .05382 Mdot
.91459 .05314 Mdot
.91475 .05299 Mdot
.91492 .05266 Mdot
.91508 .0523 Mdot
.91524 .0522 Mdot
.91542 .05237 Mdot
.9156 .05239 Mdot
.91579 .05261 Mdot
.91598 .05274 Mdot
.91615 .05209 Mdot
.91632 .05186 Mdot
.91649 .05158 Mdot
.91665 .05121 Mdot
.91681 .05104 Mdot
.91699 .05122 Mdot
.91717 .05125 Mdot
.91735 .05138 Mdot
.91755 .05171 Mdot
.91773 .0511 Mdot
.91789 .05079 Mdot
.91806 .05058 Mdot
.91822 .0502 Mdot
.91839 .04998 Mdot
.91856 .05014 Mdot
.91874 .0502 Mdot
.91892 .05028 Mdot
.91913 .05086 Mdot
.9193 .05029 Mdot
.91948 .04989 Mdot
.91964 .04973 Mdot
.9198 .04934 Mdot
.91997 .04908 Mdot
.92014 .04923 Mdot
.92032 .0493 Mdot
.9205 .04936 Mdot
.92071 .04996 Mdot
.92088 .04942 Mdot
.92106 .04898 Mdot
.92122 .04885 Mdot
.92138 .04846 Mdot
.92155 .04818 Mdot
.92172 .0483 Mdot
.9219 .0484 Mdot
.92208 .04844 Mdot
.92228 .04904 Mdot
.92246 .04859 Mdot
.92264 .04808 Mdot
.9228 .04798 Mdot
.92296 .0476 Mdot
.92313 .04729 Mdot
.9233 .04737 Mdot
.92348 .0475 Mdot
.92366 .04753 Mdot
.92386 .04812 Mdot
.92404 .04777 Mdot
.92421 .0472 Mdot
.92438 .04711 Mdot
.92454 .04673 Mdot
.9247 .0464 Mdot
.92487 .04644 Mdot
.92505 .04658 Mdot
.92523 .0466 Mdot
.92543 .04714 Mdot
.92561 .04692 Mdot
.92578 .04628 Mdot
.92595 .04618 Mdot
.92612 .04581 Mdot
.92627 .04546 Mdot
.92644 .04544 Mdot
.92662 .04561 Mdot
.9268 .04562 Mdot
.927 .04605 Mdot
.92718 .04596 Mdot
.92736 .04528 Mdot
.92752 .04515 Mdot
.92769 .0448 Mdot
.92785 .04445 Mdot
.92801 .04436 Mdot
.92819 .04454 Mdot
.92837 .04456 Mdot
.92856 .04481 Mdot
.92875 .04488 Mdot
.92892 .04423 Mdot
.92909 .04402 Mdot
.92926 .04372 Mdot
.92942 .04336 Mdot
.92958 .04321 Mdot
.92976 .04338 Mdot
.92994 .04342 Mdot
.93012 .04356 Mdot
.93032 .04383 Mdot
.93049 .04321 Mdot
.93066 .04292 Mdot
.93083 .0427 Mdot
.93099 .04233 Mdot
.93115 .04211 Mdot
.93133 .04228 Mdot
.93151 .04233 Mdot
.93169 .04242 Mdot
.93189 .04294 Mdot
.93207 .04236 Mdot
.93224 .04198 Mdot
.93241 .04181 Mdot
.93257 .04142 Mdot
.93273 .04117 Mdot
.93291 .04132 Mdot
.93309 .04139 Mdot
.93327 .04146 Mdot
.93347 .04205 Mdot
.93365 .0415 Mdot
.93382 .04107 Mdot
.93399 .04093 Mdot
.93415 .04054 Mdot
.93431 .04026 Mdot
.93449 .04039 Mdot
.93467 .04048 Mdot
.93485 .04053 Mdot
.93505 .04113 Mdot
.93523 .04065 Mdot
.9354 .04017 Mdot
.93557 .04006 Mdot
.93573 .03968 Mdot
.93589 .03937 Mdot
.93607 .03946 Mdot
.93625 .03959 Mdot
.93643 .03962 Mdot
.93663 .04022 Mdot
.93681 .03985 Mdot
.93698 .03929 Mdot
.93715 .0392 Mdot
.93731 .03882 Mdot
.93747 .03849 Mdot
.93764 .03854 Mdot
.93782 .03868 Mdot
.938 .0387 Mdot
.9382 .03925 Mdot
.93838 .039 Mdot
.93855 .03838 Mdot
.93872 .03828 Mdot
.93888 .03791 Mdot
.93904 .03757 Mdot
.93921 .03756 Mdot
.93939 .03772 Mdot
.93957 .03773 Mdot
.93977 .03822 Mdot
.93995 .0381 Mdot
.94012 .03742 Mdot
.94029 .03729 Mdot
.94046 .03694 Mdot
.94062 .03658 Mdot
.94078 .03652 Mdot
.94096 .03669 Mdot
.94114 .03671 Mdot
.94133 .037 Mdot
.94152 .03703 Mdot
.94169 .03636 Mdot
.94186 .03618 Mdot
.94203 .03587 Mdot
.94219 .03551 Mdot
.94235 .03537 Mdot
.94253 .03555 Mdot
.9427 .03558 Mdot
.94289 .03575 Mdot
.94308 .03596 Mdot
.94326 .03533 Mdot
.94343 .03506 Mdot
.9436 .03481 Mdot
.94376 .03445 Mdot
.94392 .03425 Mdot
.9441 .03442 Mdot
.94428 .03446 Mdot
.94446 .03457 Mdot
.94466 .03501 Mdot
.94484 .03442 Mdot
.94501 .03406 Mdot
.94518 .03388 Mdot
.94533 .0335 Mdot
.9455 .03326 Mdot
.94567 .03341 Mdot
.94585 .03348 Mdot
.94604 .03354 Mdot
.94624 .03413 Mdot
.94642 .03356 Mdot
.94659 .03315 Mdot
.94676 .033 Mdot
.94692 .03261 Mdot
.94708 .03234 Mdot
.94725 .03248 Mdot
.94743 .03256 Mdot
.94762 .03261 Mdot
.94782 .03321 Mdot
.948 .03271 Mdot
.94817 .03224 Mdot
.94834 .03213 Mdot
.9485 .03175 Mdot
.94866 .03145 Mdot
.94883 .03155 Mdot
.94901 .03166 Mdot
.94919 .03169 Mdot
.94939 .0323 Mdot
.94957 .0319 Mdot
.94974 .03136 Mdot
.94991 .03127 Mdot
.95007 .03089 Mdot
.95023 .03056 Mdot
.95041 .03062 Mdot
.95059 .03076 Mdot
.95077 .03078 Mdot
.95097 .03135 Mdot
.95115 .03106 Mdot
.95132 .03046 Mdot
.95149 .03037 Mdot
.95165 .02999 Mdot
.95181 .02965 Mdot
.95198 .02965 Mdot
.95216 .02981 Mdot
.95233 .02983 Mdot
.95253 .03033 Mdot
.95272 .03018 Mdot
.95289 .02951 Mdot
.95306 .02939 Mdot
.95322 .02904 Mdot
.95338 .02868 Mdot
.95355 .02863 Mdot
.95373 .0288 Mdot
.9539 .02881 Mdot
.9541 .02914 Mdot
.95428 .02913 Mdot
.95446 .02846 Mdot
.95463 .02829 Mdot
.95479 .02798 Mdot
.95495 .02761 Mdot
.95512 .0275 Mdot
.9553 .02767 Mdot
.95547 .0277 Mdot
.95566 .02788 Mdot
.95585 .02806 Mdot
.95603 .02742 Mdot
.9562 .02717 Mdot
.95637 .0269 Mdot
.95652 .02654 Mdot
.95669 .02635 Mdot
.95687 .02652 Mdot
.95704 .02657 Mdot
.95723 .02668 Mdot
.95742 .02707 Mdot
.9576 .02647 Mdot
.95777 .02613 Mdot
.95794 .02594 Mdot
.9581 .02556 Mdot
.95826 .02532 Mdot
.95844 .02548 Mdot
.95862 .02554 Mdot
.9588 .02562 Mdot
.959 .0262 Mdot
.95918 .02563 Mdot
.95935 .02522 Mdot
.95952 .02507 Mdot
.95968 .02468 Mdot
.95984 .02441 Mdot
.96002 .02456 Mdot
.9602 .02464 Mdot
.96038 .02469 Mdot
.96058 .02529 Mdot
.96076 .02477 Mdot
.96093 .02431 Mdot
.9611 .02419 Mdot
.96126 .02381 Mdot
.96142 .02351 Mdot
.9616 .02363 Mdot
.96178 .02373 Mdot
.96196 .02377 Mdot
.96216 .02437 Mdot
.96234 .02395 Mdot
.96251 .02343 Mdot
.96268 .02333 Mdot
.96284 .02295 Mdot
.963 .02263 Mdot
.96317 .0227 Mdot
.96335 .02283 Mdot
.96353 .02285 Mdot
.96373 .02343 Mdot
.96391 .02312 Mdot
.96409 .02253 Mdot
.96425 .02244 Mdot
.96442 .02207 Mdot
.96458 .02173 Mdot
.96475 .02174 Mdot
.96493 .0219 Mdot
.9651 .02191 Mdot
.9653 .02243 Mdot
.96548 .02225 Mdot
.96566 .0216 Mdot
.96582 .02149 Mdot
.96599 .02112 Mdot
.96615 .02077 Mdot
.96632 .02073 Mdot
.9665 .0209 Mdot
.96667 .02091 Mdot
.96687 .02128 Mdot
.96705 .02124 Mdot
.96723 .02056 Mdot
.96739 .02041 Mdot
.96756 .02008 Mdot
.96772 .01972 Mdot
.96789 .01962 Mdot
.96806 .01979 Mdot
.96824 .01981 Mdot
.96843 .02002 Mdot
.96862 .02016 Mdot
.96879 .01951 Mdot
.96896 .01928 Mdot
.96913 .019 Mdot
.96929 .01864 Mdot
.96946 .01846 Mdot
.96963 .01864 Mdot
.96981 .01867 Mdot
.97 .0188 Mdot
.97019 .01914 Mdot
.97037 .01853 Mdot
.97054 .01821 Mdot
.97071 .01801 Mdot
.97086 .01763 Mdot
.97103 .01741 Mdot
.97121 .01757 Mdot
.97138 .01763 Mdot
.97157 .0177 Mdot
.97177 .01828 Mdot
.97195 .01771 Mdot
.97212 .01732 Mdot
.97229 .01716 Mdot
.97245 .01676 Mdot
.97261 .0165 Mdot
.97279 .01665 Mdot
.97296 .01673 Mdot
.97315 .01678 Mdot
.97335 .01738 Mdot
.97353 .01685 Mdot
.9737 .0164 Mdot
.97387 .01628 Mdot
.97403 .01589 Mdot
.97419 .0156 Mdot
.97436 .01572 Mdot
.97454 .01582 Mdot
.97472 .01586 Mdot
.97493 .01646 Mdot
.97511 .01601 Mdot
.97528 .01551 Mdot
.97544 .01541 Mdot
.97561 .01503 Mdot
.97577 .01472 Mdot
.97594 .01479 Mdot
.97607 .0148 Mdot
.97619 .0148 Mdot
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[29]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00
=@3oool00`000000oooo0?ooo`3X0?ooo`00=@3oool200000=X0oooo0P00000=0?ooo`00=@3oool0
0`000000oooo0?ooo`3H0?ooo`<000003@3oool003D0oooo00<000000?ooo`3oool0eP3oool30000
00l0oooo000e0?ooo`030000003oool0oooo0=D0oooo0`00000@0?ooo`00=@3oool00`000000oooo
0?ooo`3C0?ooo`<000004P3oool003D0oooo0P00003B0?ooo`<00000503oool001@0oooo0P000003
0?ooo`050000003oool0oooo0?ooo`000000103oool500000080oooo0P00000:0?ooo`030000003o
ool0oooo0=00oooo0`00000E0?ooo`004`3oool010000000oooo0?ooo`0000030?ooo`030000003o
ool0000000L0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo
00<000000?ooo`3oool0cP3oool3000001L0oooo000E0?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool01P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0
1`3oool00`000000oooo0?ooo`3=0?ooo`8000006@3oool000d0oooo100000050?ooo`070000003o
ool0oooo0?ooo`000000oooo000000070?ooo`050000003oool0oooo0?ooo`0000000P3oool00`00
0000oooo0?ooo`070?ooo`<00000b`3oool3000001X0oooo000C0?ooo`040000003oool0oooo0000
0080oooo00D000000?ooo`3oool0oooo000000060?ooo`050000003oool0oooo0?ooo`0000000P3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0<T0oooo0`00000L0?ooo`00503oool2
000000d0oooo0P0000040?ooo`8000002@3oool200000<X0oooo0`00000M0?ooo`00=03oool20000
0<P0oooo0`00000O0?ooo`00;P3oool3000000@0oooo0P0000360?ooo`8000008@3oool003D0oooo
00<000000?ooo`3oool0``3oool300000280oooo000c0?ooo`<00000``3oool3000002@0oooo000e
0?ooo`030000003oool0oooo0<00oooo0P00000V0?ooo`00=@3oool00`000000oooo0?ooo`2n0?oo
o`<000009`3oool003D0oooo0P00002m0?ooo`<00000:@3oool003D0oooo00<000000?ooo`3oool0
^`3oool3000002X0oooo000e0?ooo`030000003oool0oooo0;T0oooo0`00000/0?ooo`00=@3oool0
0`000000oooo0?ooo`2h0?ooo`800000;P3oool003D0oooo0P00002g0?ooo`<00000;`3oool003D0
oooo00<000000?ooo`3oool0]03oool300000340oooo000e0?ooo`030000003oool0oooo0;<0oooo
0P00000c0?ooo`00=@3oool00`000000oooo0?ooo`2a0?ooo`<00000=03oool003D0oooo0P00002`
0?ooo`<00000=P3oool000L0oooo100000030?ooo`800000103oool2000000<0oooo00D000000?oo
o`3oool0oooo000000040?ooo`D000000P3oool2000000X0oooo00<000000?ooo`3oool0[P3oool3
000003L0oooo00070?ooo`030000003oool0oooo00T0oooo00@000000?ooo`3oool000000`3oool0
0`000000oooo000000070?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`070?ooo`030000003oool0oooo0:`0oooo0`00000i0?ooo`00203oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool01P3oool01@000000oooo0?ooo`3o
ool000000080oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`2[0?ooo`800000>`3o
ool000030?ooo`000000000000800000103oool00`000000oooo0?ooo`080?ooo`800000103oool0
0`000000oooo000000070?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`070?ooo`<00000Z@3oool3000003`0oooo00070?ooo`040000003oool0oooo000000T0oooo00<0
00000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool0000000H0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0Y`3oool3000003h0
oooo00080?ooo`8000002P3oool3000000`0oooo0P0000040?ooo`8000002@3oool200000:P0oooo
0P0000100?ooo`00=03oool200000:H0oooo0`0000110?ooo`00;P3oool3000000@0oooo00<00000
0?ooo`3oool0XP3oool3000004<0oooo000e0?ooo`800000XP3oool3000004@0oooo000c0?ooo`<0
0000X@3oool3000004H0oooo000e0?ooo`030000003oool0oooo09h0oooo0P0000180?ooo`00=@3o
ool00`000000oooo0?ooo`2L0?ooo`<00000B@3oool003D0oooo0P00002L0?ooo`800000B`3oool0
03D0oooo00<000000?ooo`3oool0V@3oool3000004`0oooo000e0?ooo`030000003oool0oooo09L0
oooo0`00001>0?ooo`00=@3oool00`000000oooo0?ooo`2E0?ooo`@00000C`3oool003D0oooo0P00
002E0?ooo`<00000D@3oool003D0oooo00<000000?ooo`3oool0TP3oool3000005<0oooo000e0?oo
o`030000003oool0oooo0940oooo0P00001E0?ooo`00=@3oool00`000000oooo0?ooo`2?0?ooo`<0
0000EP3oool003D0oooo0P00002>0?ooo`<00000F03oool003D0oooo00<000000?ooo`3oool0S03o
ool3000005T0oooo000C0?ooo`@000000P3oool01@000000oooo0?ooo`3oool0000000@0oooo1@00
00020?ooo`8000002P3oool00`000000oooo0?ooo`2:0?ooo`<00000F`3oool001<0oooo00<00000
0?ooo`3oool0103oool00`000000oooo000000070?ooo`050000003oool0oooo0?ooo`0000000P3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo08T0oooo0`00001L0?ooo`00503oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo0000
00020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0Q`3oool3000005h0oooo000=
0?ooo`@00000103oool00`000000oooo0?ooo`020?ooo`030000003oool0000000L0oooo00D00000
0?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo0`0000260?ooo`800000H03o
ool001<0oooo00@000000?ooo`3oool000000P3oool01@000000oooo0?ooo`3oool0000000H0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3o
ool0Q03oool300000640oooo000D0?ooo`8000003@3oool2000000@0oooo0P0000090?ooo`800000
Q03oool3000006<0oooo000d0?ooo`800000PP3oool3000006D0oooo000^0?ooo`<00000103oool2
00000800oooo0`00001V0?ooo`00=@3oool00`000000oooo0?ooo`1m0?ooo`<00000J03oool003<0
oooo0`00001n0?ooo`<00000J@3oool003D0oooo00<000000?ooo`3oool0NP3oool3000006/0oooo
000e0?ooo`800000NP3oool2000006d0oooo000e0?ooo`030000003oool0oooo07L0oooo0`00001^
0?ooo`00=@3oool00`000000oooo0?ooo`1e0?ooo`<00000L03oool003D0oooo00<000000?ooo`3o
ool0M03oool300000740oooo000e0?ooo`030000003oool0oooo0780oooo0`00001c0?ooo`00=@3o
ool200000740oooo0`00001e0?ooo`00=@3oool00`000000oooo0?ooo`1_0?ooo`<00000MP3oool0
03D0oooo00<000000?ooo`3oool0K@3oool3000007P0oooo000e0?ooo`030000003oool0oooo06`0
oooo0P00001j0?ooo`00=@3oool2000006/0oooo0`00001k0?ooo`001`3oool500000080oooo0P00
00040?ooo`8000000`3oool01@000000oooo0?ooo`3oool0000000@0oooo1@0000020?ooo`800000
2P3oool00`000000oooo0?ooo`1X0?ooo`<00000O@3oool000T0oooo00<000000?ooo`3oool01`3o
ool010000000oooo0?ooo`0000030?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0I`3oool3000007h0
oooo00090?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool00P3oool00`000000oooo
0?ooo`060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo06D0oooo0`0000200?ooo`0000<0oooo0000000000000P0000040?ooo`030000
003oool0oooo00P0oooo0P0000040?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00L0oooo0`00001T0?ooo`800000PP3oool000T0oooo
00<000000?ooo`3oool0203oool00`000000oooo0?ooo`020?ooo`050000003oool0oooo0?ooo`00
00001P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`1R0?ooo`<00000P`3oool000P0oooo0P00000:0?ooo`<00000303oool2000000@0
oooo0P0000090?ooo`800000HP3oool3000008D0oooo000d0?ooo`800000H@3oool2000008L0oooo
000^0?ooo`<00000103oool2000005h0oooo0`0000280?ooo`00=@3oool00`000000oooo0?ooo`1K
0?ooo`<00000RP3oool003<0oooo0`00001L0?ooo`<00000R`3oool003D0oooo00<000000?ooo`3o
ool0F03oool3000008d0oooo000e0?ooo`030000003oool0oooo05L0oooo0P00002?0?ooo`00=@3o
ool2000005H0oooo0`00002@0?ooo`00=@3oool00`000000oooo0?ooo`1C0?ooo`<00000TP3oool0
03D0oooo00<000000?ooo`3oool0DP3oool2000009@0oooo000e0?ooo`030000003oool0oooo0500
oooo0`00002E0?ooo`00=@3oool200000500oooo0P00002G0?ooo`00=@3oool00`000000oooo0?oo
o`1=0?ooo`<00000V03oool003D0oooo00<000000?ooo`3oool0B`3oool3000009X0oooo000e0?oo
o`030000003oool0oooo04X0oooo0P00002L0?ooo`00=@3oool2000004T0oooo0`00002M0?ooo`00
4`3oool5000000030?ooo`000000oooo0080oooo00<000000?ooo`3oool00P3oool500000080oooo
0P00000:0?ooo`030000003oool0oooo04H0oooo0`00002O0?ooo`005@3oool00`000000oooo0?oo
o`020?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000
003oool0oooo00L0oooo00<000000?ooo`3oool0A@3oool200000:40oooo000E0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool01P3oool01@000000oooo0?ooo`3oool000000080oooo
00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`130?ooo`<00000XP3oool000d0oooo1000
00040?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000001`3oool01@000000oooo0?oo
o`3oool000000080oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`110?ooo`<00000
Y03oool001D0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0oooo00@0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo0`0000100?ooo`<0
0000Y@3oool001@0oooo0P00000=0?ooo`800000103oool2000000X0oooo00<000000?ooo`3oool0
?P3oool300000:L0oooo000d0?ooo`800000?`3oool300000:P0oooo000d0?ooo`800000?@3oool3
00000:X0oooo000^0?ooo`<00000103oool2000003/0oooo0P00002/0?ooo`00=@3oool00`000000
oooo0?ooo`0h0?ooo`<00000[@3oool003<0oooo0`00000h0?ooo`<00000[`3oool003D0oooo00<0
00000?ooo`3oool0=03oool400000;00oooo000e0?ooo`800000=03oool300000;80oooo000e0?oo
o`030000003oool0oooo0340oooo0`00002d0?ooo`00=@3oool00`000000oooo0?ooo`0`0?ooo`<0
0000]@3oool003D0oooo00<000000?ooo`3oool0;P3oool300000;L0oooo000e0?ooo`800000;P3o
ool200000;T0oooo000e0?ooo`030000003oool0oooo02/0oooo0`00002j0?ooo`00=@3oool00`00
0000oooo0?ooo`0Y0?ooo`<00000_03oool003D0oooo00<000000?ooo`3oool0:03oool300000;d0
oooo000e0?ooo`030000003oool0oooo02H0oooo0`00002o0?ooo`00=@3oool2000002H0oooo0P00
00310?ooo`00503oool2000000<0oooo00D000000?ooo`3oool0oooo000000040?ooo`D000000P3o
ool2000000X0oooo00<000000?ooo`3oool08`3oool200000<<0oooo000C0?ooo`040000003oool0
oooo000000<0oooo00<000000?ooo`0000001`3oool01@000000oooo0?ooo`3oool000000080oooo
00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0Q0?ooo`<00000a03oool001H0oooo00<0
00000?ooo`3oool00P3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0200oooo0P0000360?ooo`003@3o
ool4000000<0oooo0P0000040?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0oooo
000000020?ooo`030000003oool0oooo00L0oooo0`00000N0?ooo`<00000a`3oool001@0oooo00<0
00000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool0000000H0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0703oool300000<T0
oooo000D0?ooo`<00000303oool2000000@0oooo0P0000080?ooo`<000007@3oool300000<X0oooo
000c0?ooo`030000003oool0000001/0oooo0`00003<0?ooo`00;P3oool3000000<0oooo0`00000I
0?ooo`800000cP3oool003<0oooo00<000000?ooo`000000603oool300000<l0oooo000c0?ooo`<0
00005P3oool300000=40oooo000e0?ooo`030000003oool0oooo01<0oooo0P00003C0?ooo`00=@3o
ool00`000000oooo0?ooo`0A0?ooo`<00000e03oool003D0oooo0P00000@0?ooo`<00000eP3oool0
03D0oooo00<000000?ooo`3oool03P3oool3000001d0oooo0P0000040?ooo`800000103oool20000
02D0oooo1@0000020?ooo`800000103oool2000000@0oooo0P00000R0?ooo`D000000P3oool20000
00@0oooo0P0000040?ooo`8000008`3oool4000000<0oooo0P0000040?ooo`800000103oool20000
00<0oooo000e0?ooo`030000003oool0oooo00`0oooo0`00000N0?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00000V0?ooo`050000003o
ool0oooo0?ooo`0000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
0080oooo00<000000?ooo`3oool08@3oool01@000000oooo0?ooo`3oool000000080oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo0200oooo
00<000000?ooo`3oool00`3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool000000P3oool003D0oooo00<000000?ooo`3oool02`3oool20000
02<0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`030000003o
ool0oooo02@0oooo00D000000?ooo`3oool0oooo000000020?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo
00@0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`030000003o
ool0oooo0240oooo00<000000?ooo`3oool00P3oool010000000oooo0?ooo`0000020?ooo`040000
003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool003D0oooo0P00000:0?ooo`<0
00008P3oool2000000<0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00000V
0?ooo`050000003oool0oooo0?ooo`0000000P3oool010000000oooo0?ooo`0000020?ooo`040000
003oool0oooo00000080oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`020?ooo`80
00000`3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000002@0oooo00D00000
0?ooo`3oool0oooo000000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000P3oool100000040oooo0@3oool003D0oooo00<000000?ooo`3oool01`3oool3000002@0oooo
00<000000?ooo`3oool00P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
02H0oooo00D000000?ooo`3oool0oooo000000020?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool000000P3oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo0080oooo
00<000000?ooo`3oool00P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
0280oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003o
ool0oooo00000080oooo00@000000?ooo`3oool000000P3oool003D0oooo00<000000?ooo`3oool0
1P3oool3000002D0oooo0`0000030?ooo`800000103oool2000002H0oooo0P0000040?ooo`800000
103oool2000000@0oooo0P00000S0?ooo`800000103oool3000000<0oooo0P0000040?ooo`800000
903oool2000000@0oooo0P0000040?ooo`800000103oool2000000<0oooo000e0?ooo`030000003o
ool0oooo00@0oooo0`00003Q0?ooo`00=@3oool2000000<0oooo1000003R0?ooo`00=@3oool01000
0000oooo0?ooo`3oool300000>@0oooo000e0?ooo`030000003oool0000000800000iP3oool003D0
oooo0`00003X0?ooo`00<03ooooY000000L0oooo000e0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`0:0?ooo`00=@3oool00`000000oooo0?ooo`0e
0?ooo`030000003oool0oooo03@0oooo00<000000?ooo`3oool0=03oool00`000000oooo0?ooo`0e
0?ooo`030000003oool0oooo00X0oooo000e0?ooo`030000003oool0oooo0>P0oooo000e0?ooo`03
0000003oool0oooo0>P0oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?oo
ob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?oo
ob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?oo
ob40oooo003o0?ooob40oooo003o0?ooob40oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-487.597, -3.66366*^-07, \
9.04585, 2.33674*^-09}}]
}, Open  ]],

Cell["\<\
The error in the invariant is not large, but it does show a steady and \
consistent drift. Eventually, it could be large enough to affect the fidelity \
of the solution.\
\>", "Text"],

Cell["\<\
This solves for the motion of the pendulum, constraining the motion at each \
step to lie on the invariant.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(psol\  = \ 
      First[NDSolve[{equation, \ start}, \ x, \ {t, 0, 2000}, \ 
          Method \[Rule] {Projection, \ Method \[Rule] ExplicitRungeKutta, \ 
              Invariants \[Rule] invariant}]]\)], "Input",
  CellLabel->"In[31]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"x", "\[Rule]", 
        TagBox[\(InterpolatingFunction[{{0.`, 2000.`}}, "<>"]\),
          False,
          Editable->False]}], "}"}]], "Output",
  CellLabel->"Out[31]="]
}, Open  ]],

Cell[TextData[{
  "This shows a plot of the invariant at the ends of the time steps ",
  StyleBox["NDSolve", "MR"],
  " took with the projection method."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(ts\  = \ 
      First[InterpolatingFunctionCoordinates[
          x\  /. \ psol]]; \), "\[IndentingNewLine]", 
    \(ListPlot[
      Transpose[{ts, \ \(invariant\  + \ 2\ Cos[1] /. psol\)\  /. \ 
            t \[Rule] ts}], \ PlotRange \[Rule] All]; \)}], "Input",
  CellLabel->"In[32]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.00047619 0.309017 6.62709e+014 [
[.2619 .29652 -9 -9 ]
[.2619 .29652 9 0 ]
[.5 .29652 -12 -9 ]
[.5 .29652 12 0 ]
[.7381 .29652 -12 -9 ]
[.7381 .29652 12 0 ]
[.97619 .29652 -12 -9 ]
[.97619 .29652 12 0 ]
[.01131 .04393 -43.5 -6.25 ]
[.01131 .04393 0 6.25 ]
[.01131 .17648 -43.5 -6.25 ]
[.01131 .17648 0 6.25 ]
[.01131 .44156 -37.5 -6.25 ]
[.01131 .44156 0 6.25 ]
[.01131 .5741 -37.5 -6.25 ]
[.01131 .5741 0 6.25 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.2619 .30902 m
.2619 .31527 L
s
[(500)] .2619 .29652 0 1 Mshowa
.5 .30902 m
.5 .31527 L
s
[(1000)] .5 .29652 0 1 Mshowa
.7381 .30902 m
.7381 .31527 L
s
[(1500)] .7381 .29652 0 1 Mshowa
.97619 .30902 m
.97619 .31527 L
s
[(2000)] .97619 .29652 0 1 Mshowa
.125 Mabswid
.07143 .30902 m
.07143 .31277 L
s
.11905 .30902 m
.11905 .31277 L
s
.16667 .30902 m
.16667 .31277 L
s
.21429 .30902 m
.21429 .31277 L
s
.30952 .30902 m
.30952 .31277 L
s
.35714 .30902 m
.35714 .31277 L
s
.40476 .30902 m
.40476 .31277 L
s
.45238 .30902 m
.45238 .31277 L
s
.54762 .30902 m
.54762 .31277 L
s
.59524 .30902 m
.59524 .31277 L
s
.64286 .30902 m
.64286 .31277 L
s
.69048 .30902 m
.69048 .31277 L
s
.78571 .30902 m
.78571 .31277 L
s
.83333 .30902 m
.83333 .31277 L
s
.88095 .30902 m
.88095 .31277 L
s
.92857 .30902 m
.92857 .31277 L
s
.25 Mabswid
0 .30902 m
1 .30902 L
s
.02381 .04393 m
.03006 .04393 L
s
gsave
.01131 .04393 -43.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
.02381 .17648 m
.03006 .17648 L
s
gsave
.01131 .17648 -43.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
.02381 .44156 m
.03006 .44156 L
s
gsave
.01131 .44156 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
.02381 .5741 m
.03006 .5741 L
s
gsave
.01131 .5741 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
.125 Mabswid
.02381 .07707 m
.02756 .07707 L
s
.02381 .1102 m
.02756 .1102 L
s
.02381 .14334 m
.02756 .14334 L
s
.02381 .20961 m
.02756 .20961 L
s
.02381 .24275 m
.02756 .24275 L
s
.02381 .27588 m
.02756 .27588 L
s
.02381 .34215 m
.02756 .34215 L
s
.02381 .37529 m
.02756 .37529 L
s
.02381 .40842 m
.02756 .40842 L
s
.02381 .47469 m
.02756 .47469 L
s
.02381 .50783 m
.02756 .50783 L
s
.02381 .54097 m
.02756 .54097 L
s
.02381 .0108 m
.02756 .0108 L
s
.02381 .60724 m
.02756 .60724 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.008 w
.02381 .30902 Mdot
.02389 .16187 Mdot
.02402 .30902 Mdot
.02414 .16187 Mdot
.02428 .30902 Mdot
.02441 .30902 Mdot
.02454 .30902 Mdot
.02469 .30902 Mdot
.02482 .01472 Mdot
.02497 .30902 Mdot
.02512 .16187 Mdot
.02529 .30902 Mdot
.02546 .30902 Mdot
.02564 .16187 Mdot
.02582 .30902 Mdot
.02599 .30902 Mdot
.02616 .30902 Mdot
.02632 .30902 Mdot
.02648 .30902 Mdot
.02664 .45617 Mdot
.02682 .30902 Mdot
.02699 .30902 Mdot
.02717 .16187 Mdot
.02737 .45617 Mdot
.02755 .30902 Mdot
.02772 .30902 Mdot
.02789 .30902 Mdot
.02805 .30902 Mdot
.02821 .16187 Mdot
.02839 .16187 Mdot
.02857 .30902 Mdot
.02875 .30902 Mdot
.02895 .16187 Mdot
.02913 .30902 Mdot
.0293 .30902 Mdot
.02947 .30902 Mdot
.02963 .30902 Mdot
.02979 .30902 Mdot
.02996 .45617 Mdot
.03014 .30902 Mdot
.03032 .45617 Mdot
.03052 .30902 Mdot
.0307 .30902 Mdot
.03088 .45617 Mdot
.03104 .30902 Mdot
.0312 .01472 Mdot
.03137 .30902 Mdot
.03154 .30902 Mdot
.03172 .30902 Mdot
.03189 .45617 Mdot
.03209 .30902 Mdot
.03227 .30902 Mdot
.03245 .30902 Mdot
.03261 .30902 Mdot
.03278 .30902 Mdot
.03294 .30902 Mdot
.03311 .30902 Mdot
.03328 .30902 Mdot
.03346 .30902 Mdot
.03366 .30902 Mdot
.03384 .30902 Mdot
.03402 .30902 Mdot
.03418 .30902 Mdot
.03435 .30902 Mdot
.03451 .45617 Mdot
.03468 .30902 Mdot
.03485 .16187 Mdot
.03503 .30902 Mdot
.03522 .30902 Mdot
.03541 .30902 Mdot
.03558 .16187 Mdot
.03575 .30902 Mdot
.03592 .60332 Mdot
.03608 .30902 Mdot
.03625 .30902 Mdot
.03642 .45617 Mdot
.0366 .30902 Mdot
.03678 .30902 Mdot
.03698 .30902 Mdot
.03716 .45617 Mdot
.03733 .01472 Mdot
.0375 .30902 Mdot
.03765 .60332 Mdot
.03782 .45617 Mdot
.03799 .30902 Mdot
.03817 .16187 Mdot
.03836 .45617 Mdot
.03856 .30902 Mdot
.03874 .60332 Mdot
.03891 .30902 Mdot
.03908 .30902 Mdot
.03923 .30902 Mdot
.0394 .45617 Mdot
.03958 .30902 Mdot
.03975 .30902 Mdot
.03994 .30902 Mdot
.04014 .30902 Mdot
.04032 .16187 Mdot
.04049 .45617 Mdot
.04066 .30902 Mdot
.04082 .30902 Mdot
.04098 .45617 Mdot
.04115 .16187 Mdot
.04133 .45617 Mdot
.04151 .30902 Mdot
.04171 .60332 Mdot
.04189 .30902 Mdot
.04207 .30902 Mdot
.04223 .30902 Mdot
.04239 .60332 Mdot
.04256 .30902 Mdot
.04273 .30902 Mdot
.04291 .45617 Mdot
.04309 .16187 Mdot
.04329 .30902 Mdot
.04347 .16187 Mdot
.04364 .30902 Mdot
.04381 .30902 Mdot
.04397 .30902 Mdot
.04413 .45617 Mdot
.0443 .45617 Mdot
.04448 .30902 Mdot
.04466 .30902 Mdot
.04486 .45617 Mdot
.04504 .16187 Mdot
.04522 .60332 Mdot
.04538 .30902 Mdot
.04555 .30902 Mdot
.04571 .30902 Mdot
.04587 .30902 Mdot
.04605 .30902 Mdot
.04623 .16187 Mdot
.04643 .30902 Mdot
.04661 .30902 Mdot
.04679 .30902 Mdot
.04695 .30902 Mdot
.04712 .30902 Mdot
.04728 .30902 Mdot
.04744 .16187 Mdot
.04762 .45617 Mdot
.0478 .30902 Mdot
.04799 .30902 Mdot
.04818 .30902 Mdot
.04835 .30902 Mdot
.04852 .30902 Mdot
.04869 .30902 Mdot
.04885 .30902 Mdot
.04901 .30902 Mdot
.04919 .30902 Mdot
.04937 .30902 Mdot
.04955 .16187 Mdot
.04975 .16187 Mdot
.04992 .60332 Mdot
.05009 .16187 Mdot
.05026 .30902 Mdot
.05042 .30902 Mdot
.05059 .30902 Mdot
.05076 .30902 Mdot
.05094 .30902 Mdot
.05112 .30902 Mdot
.05132 .01472 Mdot
.0515 .45617 Mdot
.05167 .30902 Mdot
.05184 .30902 Mdot
.052 .30902 Mdot
.05216 .30902 Mdot
.05234 .16187 Mdot
.05252 .16187 Mdot
.0527 .30902 Mdot
.0529 .45617 Mdot
.05308 .60332 Mdot
.05325 .30902 Mdot
.05342 .45617 Mdot
.05358 .01472 Mdot
.05374 .30902 Mdot
.05392 .30902 Mdot
.0541 .30902 Mdot
.05428 .45617 Mdot
.05448 .30902 Mdot
.05466 .30902 Mdot
.05483 .30902 Mdot
.055 .30902 Mdot
.05516 .30902 Mdot
.05532 .45617 Mdot
.0555 .30902 Mdot
.05568 .16187 Mdot
.05586 .30902 Mdot
.05606 .45617 Mdot
.05624 .16187 Mdot
.05641 .01472 Mdot
.05658 .30902 Mdot
.05674 .45617 Mdot
.0569 .30902 Mdot
.05707 .30902 Mdot
.05725 .30902 Mdot
.05743 .45617 Mdot
.05763 .30902 Mdot
.05781 .30902 Mdot
.05798 .30902 Mdot
.05815 .30902 Mdot
.05831 .30902 Mdot
.05847 .30902 Mdot
.05864 .45617 Mdot
.05882 .45617 Mdot
.059 .30902 Mdot
.0592 .30902 Mdot
.05938 .30902 Mdot
.05955 .30902 Mdot
.05972 .30902 Mdot
.05989 .30902 Mdot
.06005 .30902 Mdot
.06021 .30902 Mdot
.06039 .30902 Mdot
.06057 .30902 Mdot
.06076 .16187 Mdot
.06095 .30902 Mdot
.06112 .30902 Mdot
.06129 .30902 Mdot
.06146 .30902 Mdot
.06162 .30902 Mdot
.06178 .45617 Mdot
.06196 .30902 Mdot
.06214 .30902 Mdot
.06232 .30902 Mdot
.06252 .30902 Mdot
.06269 .30902 Mdot
.06286 .60332 Mdot
.06303 .16187 Mdot
.06319 .01472 Mdot
.06335 .30902 Mdot
.06353 .30902 Mdot
.06371 .30902 Mdot
.06389 .30902 Mdot
.06409 .30902 Mdot
.06427 .30902 Mdot
.06444 .01472 Mdot
.06461 .30902 Mdot
.06477 .01472 Mdot
.06493 .60332 Mdot
.06511 .30902 Mdot
.06528 .30902 Mdot
.06547 .30902 Mdot
.06567 .30902 Mdot
.06585 .45617 Mdot
.06602 .30902 Mdot
.06619 .30902 Mdot
.06635 .30902 Mdot
.06651 .01472 Mdot
.06669 .30902 Mdot
.06686 .30902 Mdot
.06705 .30902 Mdot
.06725 .30902 Mdot
.06743 .45617 Mdot
.0676 .30902 Mdot
.06777 .16187 Mdot
.06793 .30902 Mdot
.06809 .16187 Mdot
.06826 .45617 Mdot
.06844 .45617 Mdot
.06862 .30902 Mdot
.06882 .30902 Mdot
.069 .30902 Mdot
.06918 .30902 Mdot
.06934 .30902 Mdot
.0695 .60332 Mdot
.06967 .45617 Mdot
.06984 .16187 Mdot
.07002 .30902 Mdot
.0702 .16187 Mdot
.0704 .45617 Mdot
.07058 .30902 Mdot
.07075 .30902 Mdot
.07092 .16187 Mdot
.07108 .30902 Mdot
.07124 .30902 Mdot
.07141 .30902 Mdot
.07159 .30902 Mdot
.07177 .30902 Mdot
.07197 .45617 Mdot
.07215 .30902 Mdot
.07232 .30902 Mdot
.07249 .30902 Mdot
.07265 .30902 Mdot
.07281 .30902 Mdot
.07298 .45617 Mdot
.07316 .16187 Mdot
.07334 .45617 Mdot
.07353 .16187 Mdot
.07372 .30902 Mdot
.07389 .30902 Mdot
.07406 .30902 Mdot
.07422 .30902 Mdot
.07438 .30902 Mdot
.07455 .45617 Mdot
.07473 .45617 Mdot
.0749 .45617 Mdot
.07509 .30902 Mdot
.07528 .45617 Mdot
.07546 .30902 Mdot
.07563 .16187 Mdot
.0758 .30902 Mdot
.07595 .30902 Mdot
.07612 .30902 Mdot
.0763 .30902 Mdot
.07647 .16187 Mdot
.07666 .45617 Mdot
.07686 .16187 Mdot
.07703 .30902 Mdot
.0772 .30902 Mdot
.07737 .30902 Mdot
.07753 .30902 Mdot
.07769 .30902 Mdot
.07787 .30902 Mdot
.07805 .30902 Mdot
.07823 .30902 Mdot
.07843 .30902 Mdot
.07861 .30902 Mdot
.07878 .30902 Mdot
.07895 .16187 Mdot
.07911 .16187 Mdot
.07928 .30902 Mdot
.07945 .30902 Mdot
.07963 .45617 Mdot
.07981 .45617 Mdot
.08001 .30902 Mdot
.08019 .30902 Mdot
.08036 .30902 Mdot
.08053 .30902 Mdot
.08069 .30902 Mdot
.08085 .30902 Mdot
.08103 .45617 Mdot
.08121 .45617 Mdot
.08139 .16187 Mdot
.08159 .45617 Mdot
.08177 .30902 Mdot
.08194 .30902 Mdot
.08211 .01472 Mdot
.08227 .45617 Mdot
.08243 .45617 Mdot
.0826 .45617 Mdot
.08279 .30902 Mdot
.08296 .30902 Mdot
.08316 .30902 Mdot
.08334 .45617 Mdot
.08352 .30902 Mdot
.08368 .30902 Mdot
.08385 .30902 Mdot
.08401 .30902 Mdot
.08418 .16187 Mdot
.08436 .30902 Mdot
.08453 .30902 Mdot
.08473 .30902 Mdot
.08491 .30902 Mdot
.08509 .30902 Mdot
.08526 .30902 Mdot
.08542 .30902 Mdot
.08558 .30902 Mdot
.08575 .30902 Mdot
.08593 .30902 Mdot
.0861 .30902 Mdot
.0863 .16187 Mdot
.08648 .30902 Mdot
.08666 .30902 Mdot
.08682 .30902 Mdot
.08699 .30902 Mdot
.08715 .30902 Mdot
.08732 .30902 Mdot
.08749 .45617 Mdot
.08767 .30902 Mdot
.08786 .16187 Mdot
.08805 .30902 Mdot
.08823 .30902 Mdot
.08839 .30902 Mdot
.08856 .30902 Mdot
.08872 .30902 Mdot
.08889 .16187 Mdot
.08906 .45617 Mdot
.08924 .16187 Mdot
.08943 .30902 Mdot
.08962 .30902 Mdot
.0898 .30902 Mdot
.08997 .16187 Mdot
.09014 .30902 Mdot
.0903 .30902 Mdot
.09046 .30902 Mdot
.09064 .30902 Mdot
.09081 .30902 Mdot
.091 .16187 Mdot
.0912 .30902 Mdot
.09138 .30902 Mdot
.09155 .30902 Mdot
.09172 .30902 Mdot
.09188 .30902 Mdot
.09204 .30902 Mdot
.09222 .30902 Mdot
.0924 .16187 Mdot
.09258 .30902 Mdot
.09278 .45617 Mdot
.09296 .30902 Mdot
.09313 .30902 Mdot
.0933 .30902 Mdot
.09346 .45617 Mdot
.09362 .16187 Mdot
.0938 .45617 Mdot
.09398 .16187 Mdot
.09416 .16187 Mdot
.09436 .16187 Mdot
.09454 .30902 Mdot
.09471 .16187 Mdot
.09488 .30902 Mdot
.09504 .30902 Mdot
.0952 .30902 Mdot
.09537 .30902 Mdot
.09555 .30902 Mdot
.09573 .30902 Mdot
.09593 .16187 Mdot
.09611 .30902 Mdot
.09628 .30902 Mdot
.09645 .30902 Mdot
.09661 .30902 Mdot
.09677 .16187 Mdot
.09694 .30902 Mdot
.09712 .45617 Mdot
.0973 .16187 Mdot
.0975 .30902 Mdot
.09768 .45617 Mdot
.09786 .45617 Mdot
.09802 .30902 Mdot
.09819 .30902 Mdot
.09835 .30902 Mdot
.09852 .30902 Mdot
.09869 .30902 Mdot
.09887 .30902 Mdot
.09907 .30902 Mdot
.09925 .30902 Mdot
.09943 .30902 Mdot
.09959 .30902 Mdot
.09976 .30902 Mdot
.09992 .30902 Mdot
.10009 .16187 Mdot
.10026 .30902 Mdot
.10044 .30902 Mdot
.10063 .30902 Mdot
.10082 .45617 Mdot
.10099 .01472 Mdot
.10116 .30902 Mdot
.10133 .30902 Mdot
.10149 .30902 Mdot
.10165 .16187 Mdot
.10183 .30902 Mdot
.10201 .30902 Mdot
.10219 .30902 Mdot
.10239 .01472 Mdot
.10256 .45617 Mdot
.10273 .16187 Mdot
.1029 .30902 Mdot
.10306 .30902 Mdot
.10323 .30902 Mdot
.1034 .16187 Mdot
.10358 .30902 Mdot
.10376 .30902 Mdot
.10397 .30902 Mdot
.10414 .30902 Mdot
.10431 .30902 Mdot
.10448 .30902 Mdot
.10464 .30902 Mdot
.10481 .30902 Mdot
.10498 .30902 Mdot
.10516 .45617 Mdot
.10534 .30902 Mdot
.10554 .30902 Mdot
.10572 .16187 Mdot
.10589 .30902 Mdot
.10606 .30902 Mdot
.10622 .30902 Mdot
.10639 .45617 Mdot
.10656 .45617 Mdot
.10674 .16187 Mdot
.10692 .60332 Mdot
.10712 .16187 Mdot
.1073 .30902 Mdot
.10747 .30902 Mdot
.10764 .30902 Mdot
.1078 .30902 Mdot
.10796 .30902 Mdot
.10814 .45617 Mdot
.10832 .30902 Mdot
.1085 .30902 Mdot
.1087 .60332 Mdot
.10888 .30902 Mdot
.10905 .30902 Mdot
.10922 .30902 Mdot
.10938 .30902 Mdot
.10954 .30902 Mdot
.10971 .30902 Mdot
.10989 .30902 Mdot
.11007 .30902 Mdot
.11027 .30902 Mdot
.11045 .30902 Mdot
.11062 .30902 Mdot
.11079 .30902 Mdot
.11095 .30902 Mdot
.11111 .30902 Mdot
.11128 .30902 Mdot
.11146 .30902 Mdot
.11164 .30902 Mdot
.11184 .30902 Mdot
.11202 .30902 Mdot
.1122 .30902 Mdot
.11236 .30902 Mdot
.11253 .16187 Mdot
.11269 .30902 Mdot
.11286 .30902 Mdot
.11303 .30902 Mdot
.11321 .60332 Mdot
.1134 .30902 Mdot
.11359 .01472 Mdot
.11376 .30902 Mdot
.11393 .30902 Mdot
.1141 .30902 Mdot
.11426 .30902 Mdot
.11442 .30902 Mdot
.1146 .30902 Mdot
.11478 .30902 Mdot
.11496 .30902 Mdot
.11516 .16187 Mdot
.11533 .30902 Mdot
.1155 .30902 Mdot
.11567 .30902 Mdot
.11583 .60332 Mdot
.11599 .30902 Mdot
.11617 .45617 Mdot
.11635 .45617 Mdot
.11653 .16187 Mdot
.11673 .45617 Mdot
.11691 .30902 Mdot
.11708 .30902 Mdot
.11725 .30902 Mdot
.11741 .30902 Mdot
.11757 .30902 Mdot
.11775 .30902 Mdot
.11793 .16187 Mdot
.11811 .16187 Mdot
.11831 .30902 Mdot
.11849 .30902 Mdot
.11866 .30902 Mdot
.11883 .30902 Mdot
.11899 .30902 Mdot
.11915 .30902 Mdot
.11933 .30902 Mdot
.11951 .30902 Mdot
.11969 .30902 Mdot
.11989 .30902 Mdot
.12007 .30902 Mdot
.12024 .45617 Mdot
.12041 .30902 Mdot
.12057 .16187 Mdot
.12073 .45617 Mdot
.1209 .30902 Mdot
.12109 .30902 Mdot
.12126 .30902 Mdot
.12147 .30902 Mdot
.12165 .30902 Mdot
.12182 .30902 Mdot
.12198 .30902 Mdot
.12215 .60332 Mdot
.12231 .16187 Mdot
.12248 .30902 Mdot
.12266 .30902 Mdot
.12284 .30902 Mdot
.12304 .16187 Mdot
.12322 .30902 Mdot
.12339 .45617 Mdot
.12356 .30902 Mdot
.12372 .30902 Mdot
.12388 .45617 Mdot
.12405 .45617 Mdot
.12423 .30902 Mdot
.12441 .30902 Mdot
.12461 .30902 Mdot
.12479 .16187 Mdot
.12496 .45617 Mdot
.12513 .30902 Mdot
.1253 .45617 Mdot
.12545 .30902 Mdot
.12562 .30902 Mdot
.1258 .30902 Mdot
.12598 .30902 Mdot
.12617 .16187 Mdot
.12636 .30902 Mdot
.12653 .30902 Mdot
.1267 .30902 Mdot
.12687 .16187 Mdot
.12702 .30902 Mdot
.12719 .30902 Mdot
.12737 .30902 Mdot
.12754 .30902 Mdot
.12773 .45617 Mdot
.12792 .30902 Mdot
.1281 .30902 Mdot
.12827 .30902 Mdot
.12844 .30902 Mdot
.1286 .30902 Mdot
.12876 .30902 Mdot
.12894 .30902 Mdot
.12911 .30902 Mdot
.1293 .30902 Mdot
.1295 .30902 Mdot
.12967 .45617 Mdot
.12984 .30902 Mdot
.13001 .30902 Mdot
.13017 .30902 Mdot
.13034 .45617 Mdot
.13051 .30902 Mdot
.13069 .30902 Mdot
.13087 .30902 Mdot
.13108 .30902 Mdot
.13125 .30902 Mdot
.13142 .30902 Mdot
.13159 .30902 Mdot
.13175 .45617 Mdot
.13192 .01472 Mdot
.13209 .30902 Mdot
.13227 .30902 Mdot
.13245 .30902 Mdot
.13265 .30902 Mdot
.13283 .30902 Mdot
.13301 .30902 Mdot
.13317 .30902 Mdot
.13333 .30902 Mdot
.1335 .45617 Mdot
.13367 .16187 Mdot
.13385 .30902 Mdot
.13403 .30902 Mdot
.13423 .30902 Mdot
.13441 .16187 Mdot
.13458 .30902 Mdot
.13475 .30902 Mdot
.13491 .30902 Mdot
.13507 .30902 Mdot
.13525 .16187 Mdot
.13543 .45617 Mdot
.1356 .45617 Mdot
.1358 .01472 Mdot
.13599 .30902 Mdot
.13616 .30902 Mdot
.13633 .30902 Mdot
.13649 .30902 Mdot
.13665 .30902 Mdot
.13682 .30902 Mdot
.137 .45617 Mdot
.13717 .16187 Mdot
.13737 .60332 Mdot
.13756 .30902 Mdot
.13773 .30902 Mdot
.1379 .30902 Mdot
.13806 .30902 Mdot
.13822 .30902 Mdot
.13839 .30902 Mdot
.13857 .45617 Mdot
.13874 .16187 Mdot
.13894 .30902 Mdot
.13912 .30902 Mdot
.1393 .30902 Mdot
.13947 .30902 Mdot
.13963 .30902 Mdot
.13979 .30902 Mdot
.13996 .30902 Mdot
.14014 .30902 Mdot
.14031 .16187 Mdot
.1405 .30902 Mdot
.14069 .16187 Mdot
.14087 .60332 Mdot
.14103 .30902 Mdot
.1412 .30902 Mdot
.14136 .30902 Mdot
.14153 .30902 Mdot
.1417 .45617 Mdot
.14188 .45617 Mdot
.14207 .30902 Mdot
.14226 .45617 Mdot
.14244 .45617 Mdot
.14261 .30902 Mdot
.14278 .30902 Mdot
.14294 .60332 Mdot
.1431 .16187 Mdot
.14328 .30902 Mdot
.14346 .16187 Mdot
.14364 .45617 Mdot
.14384 .30902 Mdot
.14402 .01472 Mdot
.14419 .30902 Mdot
.14436 .30902 Mdot
.14452 .30902 Mdot
.14468 .16187 Mdot
.14486 .30902 Mdot
.14504 .30902 Mdot
.14522 .30902 Mdot
.14542 .30902 Mdot
.1456 .30902 Mdot
.14577 .01472 Mdot
.14594 .16187 Mdot
.1461 .30902 Mdot
.14626 .30902 Mdot
.14644 .16187 Mdot
.14662 .30902 Mdot
.1468 .30902 Mdot
.147 .30902 Mdot
.14718 .30902 Mdot
.14735 .30902 Mdot
.14752 .16187 Mdot
.14768 .30902 Mdot
.14784 .16187 Mdot
.14801 .45617 Mdot
.14819 .45617 Mdot
.14837 .30902 Mdot
.14857 .30902 Mdot
.14875 .30902 Mdot
.14892 .45617 Mdot
.14909 .30902 Mdot
.14925 .30902 Mdot
.14941 .30902 Mdot
.14959 .30902 Mdot
.14977 .30902 Mdot
.14994 .30902 Mdot
.15014 .30902 Mdot
.15032 .30902 Mdot
.1505 .30902 Mdot
.15066 .30902 Mdot
.15083 .30902 Mdot
.15099 .30902 Mdot
.15116 .45617 Mdot
.15134 .30902 Mdot
.15151 .45617 Mdot
.15171 .30902 Mdot
.15189 .30902 Mdot
.15207 .30902 Mdot
.15223 .30902 Mdot
.1524 .30902 Mdot
.15256 .30902 Mdot
.15273 .45617 Mdot
.1529 .16187 Mdot
.15308 .30902 Mdot
.15327 .30902 Mdot
.15346 .30902 Mdot
.15363 .30902 Mdot
.1538 .30902 Mdot
.15397 .30902 Mdot
.15413 .16187 Mdot
.1543 .30902 Mdot
.15447 .30902 Mdot
.15465 .30902 Mdot
.15484 .16187 Mdot
.15503 .30902 Mdot
.15521 .30902 Mdot
.15538 .30902 Mdot
.15554 .30902 Mdot
.1557 .45617 Mdot
.15587 .30902 Mdot
.15604 .45617 Mdot
.15622 .30902 Mdot
.15641 .30902 Mdot
.15661 .30902 Mdot
.15678 .30902 Mdot
.15696 .30902 Mdot
.15712 .30902 Mdot
.15728 .30902 Mdot
.15745 .30902 Mdot
.15762 .16187 Mdot
.1578 .16187 Mdot
.15798 .30902 Mdot
.15819 .45617 Mdot
.15837 .01472 Mdot
.15854 .30902 Mdot
.15871 .30902 Mdot
.15886 .30902 Mdot
.15903 .30902 Mdot
.1592 .30902 Mdot
.15938 .30902 Mdot
.15956 .30902 Mdot
.15976 .30902 Mdot
.15994 .30902 Mdot
.16012 .45617 Mdot
.16028 .16187 Mdot
.16044 .60332 Mdot
.16061 .60332 Mdot
.16078 .30902 Mdot
.16096 .45617 Mdot
.16114 .16187 Mdot
.16134 .30902 Mdot
.16152 .30902 Mdot
.16169 .30902 Mdot
.16186 .30902 Mdot
.16202 .30902 Mdot
.16218 .30902 Mdot
.16235 .30902 Mdot
.16253 .30902 Mdot
.16271 .30902 Mdot
.16291 .45617 Mdot
.16309 .30902 Mdot
.16327 .30902 Mdot
.16343 .30902 Mdot
.1636 .30902 Mdot
.16376 .01472 Mdot
.16393 .60332 Mdot
.1641 .16187 Mdot
.16428 .30902 Mdot
.16448 .45617 Mdot
.16466 .30902 Mdot
.16484 .30902 Mdot
.165 .30902 Mdot
.16517 .30902 Mdot
.16533 .45617 Mdot
.1655 .30902 Mdot
.16567 .30902 Mdot
.16585 .30902 Mdot
.16604 .16187 Mdot
.16623 .60332 Mdot
.16641 .16187 Mdot
.16657 .30902 Mdot
.16674 .30902 Mdot
.1669 .30902 Mdot
.16707 .30902 Mdot
.16724 .16187 Mdot
.16742 .45617 Mdot
.16761 .30902 Mdot
.1678 .30902 Mdot
.16797 .30902 Mdot
.16814 .30902 Mdot
.16831 .30902 Mdot
.16847 .30902 Mdot
.16864 .30902 Mdot
.16881 .16187 Mdot
.16899 .45617 Mdot
.16917 .30902 Mdot
.16937 .16187 Mdot
.16955 .30902 Mdot
.16972 .30902 Mdot
.16989 .30902 Mdot
.17005 .30902 Mdot
.17021 .30902 Mdot
.17039 .30902 Mdot
.17057 .30902 Mdot
.17075 .16187 Mdot
.17095 .45617 Mdot
.17113 .01472 Mdot
.1713 .30902 Mdot
.17147 .30902 Mdot
.17163 .30902 Mdot
.17179 .30902 Mdot
.17197 .30902 Mdot
.17215 .30902 Mdot
.17233 .45617 Mdot
.17253 .30902 Mdot
.17271 .30902 Mdot
.17288 .30902 Mdot
.17305 .30902 Mdot
.17321 .30902 Mdot
.17337 .30902 Mdot
.17355 .30902 Mdot
.17373 .16187 Mdot
.17391 .30902 Mdot
.17411 .30902 Mdot
.17429 .30902 Mdot
.17446 .30902 Mdot
.17463 .30902 Mdot
.17479 .30902 Mdot
.17495 .30902 Mdot
.17512 .30902 Mdot
.1753 .30902 Mdot
.17548 .30902 Mdot
.17568 .45617 Mdot
.17586 .30902 Mdot
.17603 .16187 Mdot
.1762 .30902 Mdot
.17636 .30902 Mdot
.17652 .30902 Mdot
.17669 .30902 Mdot
.17687 .45617 Mdot
.17705 .30902 Mdot
.17725 .30902 Mdot
.17743 .30902 Mdot
.1776 .16187 Mdot
.17777 .30902 Mdot
.17794 .01472 Mdot
.1781 .30902 Mdot
.17826 .30902 Mdot
.17844 .45617 Mdot
.17862 .30902 Mdot
.17881 .30902 Mdot
.179 .30902 Mdot
.17917 .30902 Mdot
.17934 .30902 Mdot
.17951 .30902 Mdot
.17967 .30902 Mdot
.17983 .45617 Mdot
.18001 .30902 Mdot
.18019 .30902 Mdot
.18037 .45617 Mdot
.18057 .30902 Mdot
.18074 .30902 Mdot
.18091 .30902 Mdot
.18108 .60332 Mdot
.18124 .45617 Mdot
.1814 .30902 Mdot
.18158 .45617 Mdot
.18176 .30902 Mdot
.18194 .30902 Mdot
.18214 .30902 Mdot
.18232 .30902 Mdot
.18249 .30902 Mdot
.18266 .30902 Mdot
.18281 .30902 Mdot
.18298 .45617 Mdot
.18315 .30902 Mdot
.18333 .30902 Mdot
.18352 .30902 Mdot
.18372 .30902 Mdot
.1839 .30902 Mdot
.18407 .30902 Mdot
.18424 .45617 Mdot
.1844 .30902 Mdot
.18456 .30902 Mdot
.18474 .30902 Mdot
.18491 .30902 Mdot
.1851 .30902 Mdot
.1853 .30902 Mdot
.18548 .30902 Mdot
.18565 .30902 Mdot
.18582 .30902 Mdot
.18598 .30902 Mdot
.18614 .30902 Mdot
.18631 .45617 Mdot
.18649 .30902 Mdot
.18667 .30902 Mdot
.18687 .30902 Mdot
.18705 .30902 Mdot
.18723 .30902 Mdot
.18739 .45617 Mdot
.18755 .30902 Mdot
.18772 .30902 Mdot
.18789 .30902 Mdot
.18807 .30902 Mdot
.18825 .30902 Mdot
.18845 .16187 Mdot
.18863 .16187 Mdot
.1888 .60332 Mdot
.18897 .30902 Mdot
.18913 .30902 Mdot
.18929 .30902 Mdot
.18946 .30902 Mdot
.18964 .30902 Mdot
.18982 .16187 Mdot
.19002 .45617 Mdot
.1902 .45617 Mdot
.19037 .30902 Mdot
.19054 .30902 Mdot
.1907 .30902 Mdot
.19086 .01472 Mdot
.19103 .45617 Mdot
.19121 .45617 Mdot
.19139 .30902 Mdot
.19158 .30902 Mdot
.19177 .30902 Mdot
.19194 .30902 Mdot
.19211 .30902 Mdot
.19227 .30902 Mdot
.19243 .30902 Mdot
.1926 .30902 Mdot
.19278 .45617 Mdot
.19295 .16187 Mdot
.19314 .30902 Mdot
.19333 .30902 Mdot
.19351 .45617 Mdot
.19368 .30902 Mdot
.19385 .30902 Mdot
.194 .30902 Mdot
.19417 .16187 Mdot
.19435 .30902 Mdot
.19452 .30902 Mdot
.19471 .30902 Mdot
.19491 .01472 Mdot
.19508 .30902 Mdot
.19525 .30902 Mdot
.19542 .30902 Mdot
.19558 .45617 Mdot
.19574 .16187 Mdot
.19592 .30902 Mdot
.1961 .45617 Mdot
.19628 .16187 Mdot
.19648 .45617 Mdot
.19666 .16187 Mdot
.19683 .30902 Mdot
.197 .30902 Mdot
.19716 .45617 Mdot
.19732 .30902 Mdot
.1975 .30902 Mdot
.19768 .16187 Mdot
.19786 .30902 Mdot
.19806 .30902 Mdot
.19824 .30902 Mdot
.19841 .30902 Mdot
.19858 .30902 Mdot
.19874 .30902 Mdot
.1989 .30902 Mdot
.19908 .30902 Mdot
.19926 .30902 Mdot
.19944 .30902 Mdot
.19964 .45617 Mdot
.19982 .30902 Mdot
.19999 .30902 Mdot
.20016 .30902 Mdot
.20032 .30902 Mdot
.20048 .30902 Mdot
.20065 .30902 Mdot
.20084 .30902 Mdot
.20101 .16187 Mdot
.20121 .30902 Mdot
.20139 .45617 Mdot
.20157 .01472 Mdot
.20173 .30902 Mdot
.2019 .30902 Mdot
.20206 .30902 Mdot
.20223 .45617 Mdot
.20241 .16187 Mdot
.20258 .30902 Mdot
.20278 .30902 Mdot
.20297 .16187 Mdot
.20314 .30902 Mdot
.20331 .30902 Mdot
.20347 .30902 Mdot
.20363 .30902 Mdot
.2038 .16187 Mdot
.20398 .30902 Mdot
.20415 .30902 Mdot
.20435 .16187 Mdot
.20453 .60332 Mdot
.20471 .16187 Mdot
.20488 .30902 Mdot
.20504 .30902 Mdot
.2052 .30902 Mdot
.20537 .16187 Mdot
.20555 .30902 Mdot
.20572 .60332 Mdot
.20591 .30902 Mdot
.2061 .16187 Mdot
.20628 .30902 Mdot
.20644 .30902 Mdot
.20661 .45617 Mdot
.20677 .30902 Mdot
.20694 .30902 Mdot
.20711 .45617 Mdot
.20729 .30902 Mdot
.20748 .45617 Mdot
.20767 .16187 Mdot
.20785 .30902 Mdot
.20802 .30902 Mdot
.20819 .30902 Mdot
.20834 .30902 Mdot
.20851 .30902 Mdot
.20869 .16187 Mdot
.20886 .45617 Mdot
.20905 .30902 Mdot
.20925 .30902 Mdot
.20943 .30902 Mdot
.2096 .30902 Mdot
.20977 .30902 Mdot
.20993 .30902 Mdot
.21009 .30902 Mdot
.21027 .30902 Mdot
.21044 .16187 Mdot
.21063 .45617 Mdot
.21083 .30902 Mdot
.21101 .30902 Mdot
.21118 .30902 Mdot
.21135 .30902 Mdot
.21151 .16187 Mdot
.21167 .30902 Mdot
.21185 .30902 Mdot
.21203 .16187 Mdot
.21221 .30902 Mdot
.21241 .30902 Mdot
.21259 .45617 Mdot
.21276 .30902 Mdot
.21293 .30902 Mdot
.21309 .30902 Mdot
.21325 .30902 Mdot
.21342 .30902 Mdot
.2136 .45617 Mdot
.21378 .30902 Mdot
.21398 .45617 Mdot
.21416 .16187 Mdot
.21434 .45617 Mdot
.2145 .30902 Mdot
.21466 .30902 Mdot
.21483 .30902 Mdot
.215 .30902 Mdot
.21518 .16187 Mdot
.21535 .30902 Mdot
.21555 .30902 Mdot
.21573 .30902 Mdot
.21591 .30902 Mdot
.21608 .30902 Mdot
.21624 .30902 Mdot
.2164 .30902 Mdot
.21657 .30902 Mdot
.21675 .16187 Mdot
.21692 .30902 Mdot
.21712 .45617 Mdot
.2173 .30902 Mdot
.21748 .30902 Mdot
.21765 .30902 Mdot
.21781 .30902 Mdot
.21797 .30902 Mdot
.21814 .30902 Mdot
.21832 .30902 Mdot
.21849 .45617 Mdot
.21868 .30902 Mdot
.21887 .30902 Mdot
.21905 .01472 Mdot
.21921 .30902 Mdot
.21938 .30902 Mdot
.21954 .30902 Mdot
.21971 .30902 Mdot
.21988 .45617 Mdot
.22006 .30902 Mdot
.22025 .30902 Mdot
.22044 .30902 Mdot
.22062 .30902 Mdot
.22079 .30902 Mdot
.22095 .30902 Mdot
.22111 .30902 Mdot
.22128 .16187 Mdot
.22145 .45617 Mdot
.22163 .45617 Mdot
.22182 .16187 Mdot
.22202 .30902 Mdot
.22219 .30902 Mdot
.22236 .30902 Mdot
.22253 .30902 Mdot
.22269 .30902 Mdot
.22286 .01472 Mdot
.22303 .45617 Mdot
.22321 .45617 Mdot
.22339 .30902 Mdot
.22359 .30902 Mdot
.22377 .30902 Mdot
.22394 .30902 Mdot
.22411 .30902 Mdot
.22427 .30902 Mdot
.22444 .30902 Mdot
.22461 .30902 Mdot
.22479 .30902 Mdot
.22497 .30902 Mdot
.22517 .45617 Mdot
.22535 .30902 Mdot
.22552 .16187 Mdot
.22569 .30902 Mdot
.22585 .30902 Mdot
.22601 .45617 Mdot
.22619 .30902 Mdot
.22637 .30902 Mdot
.22655 .30902 Mdot
.22675 .30902 Mdot
.22693 .30902 Mdot
.2271 .30902 Mdot
.22727 .30902 Mdot
.22743 .30902 Mdot
.22759 .30902 Mdot
.22776 .45617 Mdot
.22794 .30902 Mdot
.22812 .16187 Mdot
.22832 .30902 Mdot
.2285 .30902 Mdot
.22867 .30902 Mdot
.22884 .30902 Mdot
.229 .16187 Mdot
.22917 .60332 Mdot
.22933 .16187 Mdot
.22951 .30902 Mdot
.22969 .16187 Mdot
.22989 .30902 Mdot
.23007 .30902 Mdot
.23025 .16187 Mdot
.23041 .30902 Mdot
.23058 .30902 Mdot
.23074 .45617 Mdot
.23091 .16187 Mdot
.23108 .30902 Mdot
.23126 .30902 Mdot
.23145 .30902 Mdot
.23164 .30902 Mdot
.23182 .30902 Mdot
.23198 .30902 Mdot
.23215 .30902 Mdot
.23231 .30902 Mdot
.23247 .16187 Mdot
.23265 .16187 Mdot
.23283 .45617 Mdot
.23302 .45617 Mdot
.23321 .30902 Mdot
.23338 .30902 Mdot
.23355 .30902 Mdot
.23372 .30902 Mdot
.23388 .30902 Mdot
.23405 .45617 Mdot
.23422 .16187 Mdot
.2344 .16187 Mdot
.23458 .45617 Mdot
.23478 .01472 Mdot
.23496 .30902 Mdot
.23513 .16187 Mdot
.2353 .30902 Mdot
.23546 .16187 Mdot
.23562 .30902 Mdot
.2358 .30902 Mdot
.23597 .30902 Mdot
.23616 .30902 Mdot
.23636 .30902 Mdot
.23654 .16187 Mdot
.23671 .45617 Mdot
.23688 .30902 Mdot
.23704 .30902 Mdot
.2372 .30902 Mdot
.23738 .30902 Mdot
.23756 .30902 Mdot
.23774 .45617 Mdot
.23794 .45617 Mdot
.23812 .30902 Mdot
.23829 .30902 Mdot
.23846 .30902 Mdot
.23862 .60332 Mdot
.23878 .30902 Mdot
.23895 .30902 Mdot
.23914 .30902 Mdot
.23932 .45617 Mdot
.23952 .45617 Mdot
.2397 .01472 Mdot
.23987 .30902 Mdot
.24004 .30902 Mdot
.2402 .30902 Mdot
.24036 .16187 Mdot
.24053 .30902 Mdot
.24071 .30902 Mdot
.24089 .30902 Mdot
.24109 .60332 Mdot
.24127 .30902 Mdot
.24144 .30902 Mdot
.24161 .30902 Mdot
.24177 .30902 Mdot
.24193 .16187 Mdot
.2421 .30902 Mdot
.24228 .30902 Mdot
.24246 .30902 Mdot
.24266 .30902 Mdot
.24284 .45617 Mdot
.24301 .16187 Mdot
.24318 .45617 Mdot
.24335 .16187 Mdot
.2435 .30902 Mdot
.24367 .30902 Mdot
.24385 .30902 Mdot
.24403 .30902 Mdot
.24422 .30902 Mdot
.24441 .01472 Mdot
.24458 .30902 Mdot
.24475 .30902 Mdot
.24492 .30902 Mdot
.24508 .30902 Mdot
.24524 .30902 Mdot
.24542 .16187 Mdot
.2456 .45617 Mdot
.24578 .30902 Mdot
.24597 .30902 Mdot
.24615 .30902 Mdot
.24632 .30902 Mdot
.24649 .30902 Mdot
.24665 .30902 Mdot
.24681 .30902 Mdot
.24699 .16187 Mdot
.24717 .45617 Mdot
.24735 .30902 Mdot
.24755 .30902 Mdot
.24772 .30902 Mdot
.24789 .30902 Mdot
.24806 .45617 Mdot
.24822 .30902 Mdot
.24839 .16187 Mdot
.24856 .45617 Mdot
.24874 .30902 Mdot
.24892 .30902 Mdot
.24913 .30902 Mdot
.2493 .16187 Mdot
.24947 .30902 Mdot
.24964 .16187 Mdot
.2498 .30902 Mdot
.24997 .30902 Mdot
.25014 .45617 Mdot
.25032 .30902 Mdot
.2505 .45617 Mdot
.2507 .30902 Mdot
.25088 .30902 Mdot
.25106 .30902 Mdot
.25122 .30902 Mdot
.25138 .16187 Mdot
.25155 .45617 Mdot
.25172 .45617 Mdot
.2519 .30902 Mdot
.25208 .16187 Mdot
.25228 .45617 Mdot
.25246 .01472 Mdot
.25263 .45617 Mdot
.2528 .30902 Mdot
.25296 .30902 Mdot
.25312 .30902 Mdot
.2533 .60332 Mdot
.25348 .16187 Mdot
.25366 .30902 Mdot
.25386 .16187 Mdot
.25404 .30902 Mdot
.25421 .30902 Mdot
.25438 .30902 Mdot
.25454 .30902 Mdot
.2547 .30902 Mdot
.25487 .45617 Mdot
.25505 .30902 Mdot
.25523 .60332 Mdot
.25543 .30902 Mdot
.25561 .30902 Mdot
.25578 .30902 Mdot
.25595 .45617 Mdot
.25611 .30902 Mdot
.25627 .30902 Mdot
.25644 .30902 Mdot
.25662 .16187 Mdot
.2568 .30902 Mdot
.25699 .30902 Mdot
.25718 .30902 Mdot
.25735 .30902 Mdot
.25752 .30902 Mdot
.25768 .30902 Mdot
.25784 .30902 Mdot
.25801 .30902 Mdot
.25819 .30902 Mdot
.25836 .30902 Mdot
.25855 .16187 Mdot
.25874 .45617 Mdot
.25892 .30902 Mdot
.25909 .30902 Mdot
.25926 .30902 Mdot
.25941 .45617 Mdot
.25958 .16187 Mdot
.25976 .45617 Mdot
.25993 .45617 Mdot
.26012 .16187 Mdot
.26031 .30902 Mdot
.26049 .30902 Mdot
.26066 .30902 Mdot
.26083 .30902 Mdot
.26099 .30902 Mdot
.26115 .30902 Mdot
.26133 .45617 Mdot
.26151 .30902 Mdot
.26169 .30902 Mdot
.26189 .16187 Mdot
.26207 .30902 Mdot
.26224 .30902 Mdot
.26241 .30902 Mdot
.26257 .30902 Mdot
.26273 .30902 Mdot
.26291 .30902 Mdot
.26309 .45617 Mdot
.26327 .30902 Mdot
.26347 .60332 Mdot
.26365 .30902 Mdot
.26382 .30902 Mdot
.26399 .30902 Mdot
.26415 .16187 Mdot
.26431 .30902 Mdot
.26449 .30902 Mdot
.26467 .30902 Mdot
.26485 .45617 Mdot
.26505 .30902 Mdot
.26523 .30902 Mdot
.2654 .30902 Mdot
.26557 .30902 Mdot
.26573 .30902 Mdot
.26589 .30902 Mdot
.26606 .30902 Mdot
.26625 .45617 Mdot
.26642 .30902 Mdot
.26662 .45617 Mdot
.2668 .01472 Mdot
.26698 .30902 Mdot
.26714 .30902 Mdot
.26731 .30902 Mdot
.26747 .30902 Mdot
.26764 .45617 Mdot
.26782 .45617 Mdot
.268 .30902 Mdot
.2682 .30902 Mdot
.26838 .30902 Mdot
.26855 .30902 Mdot
.26872 .30902 Mdot
.26888 .45617 Mdot
.26904 .30902 Mdot
.26921 .30902 Mdot
.26939 .30902 Mdot
.26956 .30902 Mdot
.26976 .30902 Mdot
.26995 .16187 Mdot
.27012 .45617 Mdot
.27029 .30902 Mdot
.27045 .30902 Mdot
.27061 .30902 Mdot
.27078 .30902 Mdot
.27096 .16187 Mdot
.27113 .45617 Mdot
.27133 .30902 Mdot
.27151 .16187 Mdot
.27169 .30902 Mdot
.27186 .30902 Mdot
.27202 .30902 Mdot
.27218 .30902 Mdot
.27235 .30902 Mdot
.27253 .45617 Mdot
.2727 .45617 Mdot
.27289 .30902 Mdot
.27308 .16187 Mdot
.27326 .30902 Mdot
.27343 .30902 Mdot
.2736 .30902 Mdot
.27375 .45617 Mdot
.27392 .30902 Mdot
.2741 .30902 Mdot
.27427 .30902 Mdot
.27446 .16187 Mdot
.27466 .30902 Mdot
.27484 .45617 Mdot
.27501 .30902 Mdot
.27517 .30902 Mdot
.27533 .30902 Mdot
.2755 .16187 Mdot
.27567 .30902 Mdot
.27585 .16187 Mdot
.27603 .45617 Mdot
.27624 .45617 Mdot
.27642 .16187 Mdot
.27659 .30902 Mdot
.27675 .30902 Mdot
.27691 .30902 Mdot
.27708 .30902 Mdot
.27725 .16187 Mdot
.27743 .30902 Mdot
.27761 .30902 Mdot
.27781 .30902 Mdot
.27799 .01472 Mdot
.27817 .30902 Mdot
.27833 .30902 Mdot
.27849 .30902 Mdot
.27866 .30902 Mdot
.27883 .30902 Mdot
.27901 .45617 Mdot
.27919 .45617 Mdot
.27939 .45617 Mdot
.27957 .30902 Mdot
.27974 .30902 Mdot
.27991 .30902 Mdot
.28007 .30902 Mdot
.28023 .30902 Mdot
.2804 .30902 Mdot
.28059 .16187 Mdot
.28076 .30902 Mdot
.28096 .30902 Mdot
.28114 .30902 Mdot
.28132 .16187 Mdot
.28148 .30902 Mdot
.28165 .45617 Mdot
.28181 .45617 Mdot
.28198 .30902 Mdot
.28216 .45617 Mdot
.28233 .30902 Mdot
.28253 .30902 Mdot
.28271 .01472 Mdot
.28289 .45617 Mdot
.28306 .30902 Mdot
.28322 .45617 Mdot
.28338 .01472 Mdot
.28355 .30902 Mdot
.28373 .16187 Mdot
.2839 .45617 Mdot
.2841 .30902 Mdot
.28428 .30902 Mdot
.28446 .30902 Mdot
.28462 .30902 Mdot
.28479 .45617 Mdot
.28495 .30902 Mdot
.28512 .16187 Mdot
.28529 .45617 Mdot
.28547 .45617 Mdot
.28566 .30902 Mdot
.28585 .30902 Mdot
.28603 .60332 Mdot
.28619 .30902 Mdot
.28636 .30902 Mdot
.28652 .60332 Mdot
.28669 .30902 Mdot
.28686 .01472 Mdot
.28704 .30902 Mdot
.28723 .16187 Mdot
.28742 .45617 Mdot
.2876 .30902 Mdot
.28777 .30902 Mdot
.28794 .30902 Mdot
.2881 .16187 Mdot
.28826 .30902 Mdot
.28844 .30902 Mdot
.28862 .30902 Mdot
.2888 .30902 Mdot
.289 .30902 Mdot
.28918 .30902 Mdot
.28935 .30902 Mdot
.28952 .30902 Mdot
.28968 .30902 Mdot
.28984 .30902 Mdot
.29002 .30902 Mdot
.2902 .30902 Mdot
.29038 .30902 Mdot
.29058 .30902 Mdot
.29076 .30902 Mdot
.29093 .30902 Mdot
.2911 .30902 Mdot
.29126 .30902 Mdot
.29142 .01472 Mdot
.2916 .45617 Mdot
.29178 .16187 Mdot
.29196 .45617 Mdot
.29216 .30902 Mdot
.29234 .30902 Mdot
.29251 .30902 Mdot
.29268 .30902 Mdot
.29284 .30902 Mdot
.293 .16187 Mdot
.29317 .30902 Mdot
.29335 .16187 Mdot
.29353 .30902 Mdot
.29373 .16187 Mdot
.29391 .30902 Mdot
.29408 .60332 Mdot
.29425 .30902 Mdot
.29441 .30902 Mdot
.29457 .30902 Mdot
.29474 .30902 Mdot
.29492 .16187 Mdot
.2951 .30902 Mdot
.2953 .30902 Mdot
.29548 .30902 Mdot
.29566 .30902 Mdot
.29582 .16187 Mdot
.29599 .30902 Mdot
.29615 .30902 Mdot
.29632 .30902 Mdot
.29649 .30902 Mdot
.29667 .16187 Mdot
.29686 .45617 Mdot
.29705 .30902 Mdot
.29722 .30902 Mdot
.29739 .30902 Mdot
.29756 .30902 Mdot
.29772 .30902 Mdot
.29788 .30902 Mdot
.29806 .16187 Mdot
.29824 .16187 Mdot
.29843 .30902 Mdot
.29862 .30902 Mdot
.29879 .30902 Mdot
.29896 .30902 Mdot
.29913 .30902 Mdot
.29929 .30902 Mdot
.29945 .30902 Mdot
.29963 .45617 Mdot
.29981 .16187 Mdot
.29999 .30902 Mdot
.30019 .30902 Mdot
.30037 .30902 Mdot
.30054 .30902 Mdot
.30071 .30902 Mdot
.30086 .30902 Mdot
.30103 .30902 Mdot
.3012 .30902 Mdot
.30138 .60332 Mdot
.30157 .16187 Mdot
.30177 .30902 Mdot
.30195 .30902 Mdot
.30212 .16187 Mdot
.30229 .30902 Mdot
.30245 .45617 Mdot
.30261 .30902 Mdot
.30279 .45617 Mdot
.30296 .45617 Mdot
.30315 .30902 Mdot
.30335 .30902 Mdot
.30353 .30902 Mdot
.3037 .45617 Mdot
.30387 .30902 Mdot
.30403 .30902 Mdot
.30419 .45617 Mdot
.30436 .45617 Mdot
.30454 .30902 Mdot
.30472 .16187 Mdot
.30492 .30902 Mdot
.3051 .30902 Mdot
.30528 .30902 Mdot
.30544 .30902 Mdot
.3056 .01472 Mdot
.30577 .30902 Mdot
.30594 .45617 Mdot
.30612 .16187 Mdot
.3063 .30902 Mdot
.3065 .30902 Mdot
.30668 .30902 Mdot
.30685 .30902 Mdot
.30702 .30902 Mdot
.30718 .30902 Mdot
.30734 .30902 Mdot
.30751 .30902 Mdot
.30769 .30902 Mdot
.30787 .30902 Mdot
.30807 .16187 Mdot
.30825 .30902 Mdot
.30842 .16187 Mdot
.30859 .30902 Mdot
.30875 .30902 Mdot
.30891 .30902 Mdot
.30908 .30902 Mdot
.30926 .30902 Mdot
.30944 .30902 Mdot
.30963 .30902 Mdot
.30982 .30902 Mdot
.30999 .30902 Mdot
.31016 .30902 Mdot
.31033 .30902 Mdot
.31048 .30902 Mdot
.31065 .30902 Mdot
.31083 .45617 Mdot
.311 .30902 Mdot
.3112 .30902 Mdot
.31138 .30902 Mdot
.31156 .30902 Mdot
.31173 .30902 Mdot
.3119 .30902 Mdot
.31205 .30902 Mdot
.31222 .45617 Mdot
.3124 .30902 Mdot
.31257 .30902 Mdot
.31276 .30902 Mdot
.31296 .30902 Mdot
.31313 .30902 Mdot
.3133 .30902 Mdot
.31347 .30902 Mdot
.31363 .30902 Mdot
.31379 .16187 Mdot
.31397 .45617 Mdot
.31415 .30902 Mdot
.31433 .01472 Mdot
.31453 .45617 Mdot
.31471 .01472 Mdot
.31488 .30902 Mdot
.31505 .45617 Mdot
.31521 .01472 Mdot
.31537 .45617 Mdot
.31555 .16187 Mdot
.31573 .45617 Mdot
.31591 .30902 Mdot
.31611 .30902 Mdot
.31629 .30902 Mdot
.31646 .30902 Mdot
.31663 .30902 Mdot
.31679 .16187 Mdot
.31695 .45617 Mdot
.31713 .16187 Mdot
.31731 .30902 Mdot
.31749 .45617 Mdot
.31769 .01472 Mdot
.31787 .30902 Mdot
.31804 .30902 Mdot
.31821 .30902 Mdot
.31837 .30902 Mdot
.31853 .30902 Mdot
.31871 .45617 Mdot
.31889 .30902 Mdot
.31907 .16187 Mdot
.31927 .30902 Mdot
.31945 .45617 Mdot
.31962 .30902 Mdot
.31979 .30902 Mdot
.31995 .30902 Mdot
.32011 .16187 Mdot
.32028 .30902 Mdot
.32046 .16187 Mdot
.32064 .16187 Mdot
.32084 .45617 Mdot
.32102 .30902 Mdot
.32119 .30902 Mdot
.32136 .30902 Mdot
.32152 .30902 Mdot
.32168 .45617 Mdot
.32185 .45617 Mdot
.32203 .30902 Mdot
.32221 .30902 Mdot
.3224 .30902 Mdot
.32259 .30902 Mdot
.32276 .01472 Mdot
.32293 .30902 Mdot
.32309 .30902 Mdot
.32325 .30902 Mdot
.32342 .16187 Mdot
.3236 .45617 Mdot
.32377 .30902 Mdot
.32397 .30902 Mdot
.32415 .30902 Mdot
.32433 .30902 Mdot
.3245 .30902 Mdot
.32467 .30902 Mdot
.32482 .30902 Mdot
.32499 .16187 Mdot
.32517 .45617 Mdot
.32534 .30902 Mdot
.32553 .16187 Mdot
.32572 .16187 Mdot
.3259 .01472 Mdot
.32607 .30902 Mdot
.32624 .30902 Mdot
.3264 .30902 Mdot
.32656 .30902 Mdot
.32674 .30902 Mdot
.32691 .30902 Mdot
.3271 .30902 Mdot
.3273 .16187 Mdot
.32748 .01472 Mdot
.32765 .30902 Mdot
.32782 .30902 Mdot
.32798 .30902 Mdot
.32814 .30902 Mdot
.32832 .30902 Mdot
.32849 .30902 Mdot
.32868 .30902 Mdot
.32888 .30902 Mdot
.32906 .30902 Mdot
.32923 .30902 Mdot
.3294 .30902 Mdot
.32956 .45617 Mdot
.32972 .30902 Mdot
.3299 .30902 Mdot
.33007 .16187 Mdot
.33026 .30902 Mdot
.33046 .45617 Mdot
.33064 .30902 Mdot
.33081 .30902 Mdot
.33098 .30902 Mdot
.33114 .30902 Mdot
.3313 .30902 Mdot
.33147 .30902 Mdot
.33165 .30902 Mdot
.33183 .30902 Mdot
.33203 .30902 Mdot
.33221 .45617 Mdot
.33239 .60332 Mdot
.33255 .30902 Mdot
.33271 .30902 Mdot
.33288 .16187 Mdot
.33305 .30902 Mdot
.33323 .30902 Mdot
.3334 .30902 Mdot
.3336 .45617 Mdot
.33379 .30902 Mdot
.33396 .30902 Mdot
.33413 .16187 Mdot
.33429 .01472 Mdot
.33445 .30902 Mdot
.33462 .30902 Mdot
.3348 .30902 Mdot
.33497 .30902 Mdot
.33517 .30902 Mdot
.33536 .45617 Mdot
.33553 .30902 Mdot
.3357 .30902 Mdot
.33586 .30902 Mdot
.33602 .60332 Mdot
.33619 .16187 Mdot
.33637 .30902 Mdot
.33654 .30902 Mdot
.33674 .30902 Mdot
.33692 .45617 Mdot
.3371 .30902 Mdot
.33727 .30902 Mdot
.33743 .30902 Mdot
.33759 .30902 Mdot
.33776 .30902 Mdot
.33794 .16187 Mdot
.33811 .30902 Mdot
.3383 .30902 Mdot
.33849 .30902 Mdot
.33867 .30902 Mdot
.33884 .30902 Mdot
.33901 .30902 Mdot
.33916 .30902 Mdot
.33933 .45617 Mdot
.33951 .30902 Mdot
.33968 .30902 Mdot
.33987 .16187 Mdot
.34007 .45617 Mdot
.34024 .30902 Mdot
.34041 .30902 Mdot
.34058 .45617 Mdot
.34074 .30902 Mdot
.34091 .30902 Mdot
.34108 .16187 Mdot
.34126 .45617 Mdot
.34144 .30902 Mdot
.34164 .30902 Mdot
.34182 .30902 Mdot
.34199 .30902 Mdot
.34216 .30902 Mdot
.34232 .30902 Mdot
.34249 .16187 Mdot
.34266 .30902 Mdot
.34284 .30902 Mdot
.34302 .16187 Mdot
.34322 .45617 Mdot
.3434 .30902 Mdot
.34357 .30902 Mdot
.34374 .30902 Mdot
.3439 .60332 Mdot
.34406 .30902 Mdot
.34424 .30902 Mdot
.34442 .30902 Mdot
.3446 .45617 Mdot
.3448 .30902 Mdot
.34498 .01472 Mdot
.34515 .30902 Mdot
.34532 .30902 Mdot
.34548 .30902 Mdot
.34564 .16187 Mdot
.34581 .30902 Mdot
.34599 .45617 Mdot
.34617 .45617 Mdot
.34637 .30902 Mdot
.34655 .60332 Mdot
.34673 .30902 Mdot
.34689 .30902 Mdot
.34706 .30902 Mdot
.34722 .30902 Mdot
.34739 .45617 Mdot
.34756 .45617 Mdot
.34774 .16187 Mdot
.34794 .30902 Mdot
.34812 .30902 Mdot
.3483 .30902 Mdot
.34846 .30902 Mdot
.34863 .30902 Mdot
.34879 .16187 Mdot
.34896 .30902 Mdot
.34914 .16187 Mdot
.34931 .30902 Mdot
.34951 .30902 Mdot
.34969 .30902 Mdot
.34987 .30902 Mdot
.35003 .30902 Mdot
.3502 .16187 Mdot
.35036 .30902 Mdot
.35053 .30902 Mdot
.3507 .16187 Mdot
.35088 .30902 Mdot
.35107 .45617 Mdot
.35126 .30902 Mdot
.35143 .01472 Mdot
.3516 .60332 Mdot
.35177 .16187 Mdot
.35193 .30902 Mdot
.3521 .30902 Mdot
.35227 .16187 Mdot
.35245 .30902 Mdot
.35263 .45617 Mdot
.35283 .45617 Mdot
.35301 .30902 Mdot
.35318 .30902 Mdot
.35335 .30902 Mdot
.35351 .30902 Mdot
.35367 .60332 Mdot
.35385 .45617 Mdot
.35402 .16187 Mdot
.35421 .30902 Mdot
.35441 .30902 Mdot
.35459 .30902 Mdot
.35476 .30902 Mdot
.35493 .16187 Mdot
.35509 .45617 Mdot
.35525 .16187 Mdot
.35543 .30902 Mdot
.35561 .16187 Mdot
.35579 .45617 Mdot
.35599 .45617 Mdot
.35617 .30902 Mdot
.35634 .30902 Mdot
.35651 .30902 Mdot
.35667 .30902 Mdot
.35683 .30902 Mdot
.35701 .30902 Mdot
.35719 .30902 Mdot
.35737 .45617 Mdot
.35757 .45617 Mdot
.35775 .16187 Mdot
.35792 .30902 Mdot
.35809 .30902 Mdot
.35825 .30902 Mdot
.35841 .30902 Mdot
.35858 .16187 Mdot
.35876 .30902 Mdot
.35894 .16187 Mdot
.35914 .30902 Mdot
.35932 .30902 Mdot
.35949 .16187 Mdot
.35966 .30902 Mdot
.35982 .30902 Mdot
.35998 .45617 Mdot
.36015 .45617 Mdot
.36033 .30902 Mdot
.36051 .30902 Mdot
.36071 .45617 Mdot
.36089 .30902 Mdot
.36106 .30902 Mdot
.36123 .30902 Mdot
.3614 .30902 Mdot
.36156 .45617 Mdot
.36172 .30902 Mdot
.3619 .30902 Mdot
.36208 .30902 Mdot
.36227 .30902 Mdot
.36246 .30902 Mdot
.36263 .16187 Mdot
.3628 .30902 Mdot
.36297 .16187 Mdot
.36313 .30902 Mdot
.36329 .30902 Mdot
.36347 .30902 Mdot
.36365 .30902 Mdot
.36384 .45617 Mdot
.36403 .30902 Mdot
.3642 .30902 Mdot
.36437 .60332 Mdot
.36454 .30902 Mdot
.3647 .30902 Mdot
.36486 .16187 Mdot
.36504 .30902 Mdot
.36522 .30902 Mdot
.3654 .30902 Mdot
.3656 .30902 Mdot
.36577 .30902 Mdot
.36594 .30902 Mdot
.36611 .30902 Mdot
.36627 .30902 Mdot
.36644 .45617 Mdot
.36661 .16187 Mdot
.36679 .45617 Mdot
.36697 .16187 Mdot
.36717 .45617 Mdot
.36735 .30902 Mdot
.36752 .30902 Mdot
.36769 .60332 Mdot
.36785 .30902 Mdot
.36802 .30902 Mdot
.36819 .30902 Mdot
.36837 .30902 Mdot
.36855 .16187 Mdot
.36875 .45617 Mdot
.36893 .16187 Mdot
.3691 .30902 Mdot
.36927 .30902 Mdot
.36943 .30902 Mdot
.3696 .30902 Mdot
.36977 .16187 Mdot
.36995 .30902 Mdot
.37013 .01472 Mdot
.37033 .45617 Mdot
.37051 .30902 Mdot
.37068 .30902 Mdot
.37085 .30902 Mdot
.37101 .60332 Mdot
.37117 .30902 Mdot
.37135 .45617 Mdot
.37153 .30902 Mdot
.37171 .45617 Mdot
.37191 .30902 Mdot
.37209 .30902 Mdot
.37226 .16187 Mdot
.37243 .30902 Mdot
.37259 .30902 Mdot
.37275 .30902 Mdot
.37292 .01472 Mdot
.3731 .30902 Mdot
.37328 .45617 Mdot
.37348 .16187 Mdot
.37366 .30902 Mdot
.37383 .30902 Mdot
.374 .30902 Mdot
.37416 .01472 Mdot
.37432 .30902 Mdot
.37449 .30902 Mdot
.37467 .16187 Mdot
.37485 .45617 Mdot
.37504 .30902 Mdot
.37523 .30902 Mdot
.3754 .45617 Mdot
.37557 .16187 Mdot
.37574 .45617 Mdot
.37589 .30902 Mdot
.37606 .30902 Mdot
.37624 .16187 Mdot
.37642 .30902 Mdot
.37661 .30902 Mdot
.37679 .45617 Mdot
.37697 .30902 Mdot
.37714 .45617 Mdot
.37731 .16187 Mdot
.37746 .16187 Mdot
.37763 .45617 Mdot
.37781 .30902 Mdot
.37798 .16187 Mdot
.37817 .30902 Mdot
.37836 .30902 Mdot
.37854 .30902 Mdot
.37871 .30902 Mdot
.37888 .30902 Mdot
.37904 .30902 Mdot
.3792 .30902 Mdot
.37938 .30902 Mdot
.37956 .30902 Mdot
.37974 .30902 Mdot
.37994 .30902 Mdot
.38012 .30902 Mdot
.38029 .30902 Mdot
.38046 .60332 Mdot
.38062 .30902 Mdot
.38078 .30902 Mdot
.38096 .16187 Mdot
.38114 .45617 Mdot
.38132 .45617 Mdot
.38152 .30902 Mdot
.3817 .30902 Mdot
.38187 .30902 Mdot
.38204 .30902 Mdot
.3822 .45617 Mdot
.38236 .30902 Mdot
.38254 .45617 Mdot
.38272 .30902 Mdot
.3829 .30902 Mdot
.3831 .30902 Mdot
.38328 .45617 Mdot
.38345 .30902 Mdot
.38362 .16187 Mdot
.38378 .30902 Mdot
.38394 .45617 Mdot
.38411 .30902 Mdot
.3843 .30902 Mdot
.38447 .30902 Mdot
.38467 .30902 Mdot
.38485 .30902 Mdot
.38503 .30902 Mdot
.38519 .45617 Mdot
.38536 .30902 Mdot
.38552 .30902 Mdot
.38569 .16187 Mdot
.38587 .30902 Mdot
.38605 .30902 Mdot
.38625 .30902 Mdot
.38643 .45617 Mdot
.3866 .45617 Mdot
.38677 .45617 Mdot
.38693 .45617 Mdot
.38709 .16187 Mdot
.38726 .30902 Mdot
.38744 .30902 Mdot
.38762 .16187 Mdot
.38781 .30902 Mdot
.388 .30902 Mdot
.38817 .30902 Mdot
.38834 .01472 Mdot
.3885 .45617 Mdot
.38866 .30902 Mdot
.38883 .30902 Mdot
.38901 .30902 Mdot
.38919 .45617 Mdot
.38938 .30902 Mdot
.38956 .16187 Mdot
.38974 .30902 Mdot
.38991 .30902 Mdot
.39008 .30902 Mdot
.39023 .30902 Mdot
.3904 .16187 Mdot
.39058 .30902 Mdot
.39075 .30902 Mdot
.39094 .30902 Mdot
.39113 .16187 Mdot
.39131 .60332 Mdot
.39148 .30902 Mdot
.39165 .30902 Mdot
.39181 .16187 Mdot
.39197 .30902 Mdot
.39215 .16187 Mdot
.39232 .30902 Mdot
.39251 .30902 Mdot
.39271 .30902 Mdot
.39288 .30902 Mdot
.39305 .30902 Mdot
.39322 .30902 Mdot
.39338 .30902 Mdot
.39355 .45617 Mdot
.39372 .30902 Mdot
.3939 .16187 Mdot
.39408 .45617 Mdot
.39429 .30902 Mdot
.39446 .30902 Mdot
.39464 .30902 Mdot
.3948 .30902 Mdot
.39496 .16187 Mdot
.39513 .60332 Mdot
.3953 .16187 Mdot
.39548 .45617 Mdot
.39566 .30902 Mdot
.39586 .16187 Mdot
.39604 .16187 Mdot
.39622 .30902 Mdot
.39638 .45617 Mdot
.39654 .30902 Mdot
.39671 .30902 Mdot
.39688 .30902 Mdot
.39706 .16187 Mdot
.39724 .30902 Mdot
.39744 .16187 Mdot
.39762 .30902 Mdot
.39779 .30902 Mdot
.39796 .30902 Mdot
.39812 .30902 Mdot
.39828 .16187 Mdot
.39846 .45617 Mdot
.39864 .30902 Mdot
.39881 .30902 Mdot
.39901 .30902 Mdot
.39919 .16187 Mdot
.39937 .30902 Mdot
.39953 .30902 Mdot
.3997 .30902 Mdot
.39986 .30902 Mdot
.40003 .16187 Mdot
.40021 .30902 Mdot
.40038 .16187 Mdot
.40058 .30902 Mdot
.40076 .16187 Mdot
.40094 .30902 Mdot
.40111 .30902 Mdot
.40127 .30902 Mdot
.40143 .30902 Mdot
.4016 .16187 Mdot
.40178 .30902 Mdot
.40195 .45617 Mdot
.40215 .30902 Mdot
.40233 .30902 Mdot
.40251 .30902 Mdot
.40267 .60332 Mdot
.40284 .16187 Mdot
.403 .30902 Mdot
.40317 .16187 Mdot
.40334 .30902 Mdot
.40352 .45617 Mdot
.40371 .16187 Mdot
.4039 .30902 Mdot
.40408 .01472 Mdot
.40424 .30902 Mdot
.40441 .30902 Mdot
.40457 .30902 Mdot
.40474 .16187 Mdot
.40491 .16187 Mdot
.40509 .30902 Mdot
.40528 .45617 Mdot
.40547 .30902 Mdot
.40565 .30902 Mdot
.40582 .16187 Mdot
.40599 .30902 Mdot
.40615 .45617 Mdot
.40631 .30902 Mdot
.40649 .30902 Mdot
.40667 .30902 Mdot
.40685 .45617 Mdot
.40705 .30902 Mdot
.40723 .30902 Mdot
.4074 .30902 Mdot
.40757 .30902 Mdot
.40773 .16187 Mdot
.40789 .30902 Mdot
.40807 .30902 Mdot
.40825 .30902 Mdot
.40843 .45617 Mdot
.40863 .30902 Mdot
.40881 .30902 Mdot
.40898 .45617 Mdot
.40915 .30902 Mdot
.40931 .16187 Mdot
.40947 .45617 Mdot
.40965 .16187 Mdot
.40983 .16187 Mdot
.41001 .45617 Mdot
.41021 .16187 Mdot
.41039 .30902 Mdot
.41056 .45617 Mdot
.41073 .30902 Mdot
.41089 .30902 Mdot
.41105 .30902 Mdot
.41122 .45617 Mdot
.4114 .16187 Mdot
.41158 .30902 Mdot
.41178 .30902 Mdot
.41196 .01472 Mdot
.41213 .30902 Mdot
.4123 .16187 Mdot
.41246 .30902 Mdot
.41262 .60332 Mdot
.41279 .16187 Mdot
.41297 .45617 Mdot
.41315 .16187 Mdot
.41335 .30902 Mdot
.41353 .30902 Mdot
.41371 .30902 Mdot
.41387 .45617 Mdot
.41404 .01472 Mdot
.4142 .45617 Mdot
.41437 .30902 Mdot
.41454 .30902 Mdot
.41472 .30902 Mdot
.41492 .30902 Mdot
.4151 .30902 Mdot
.41527 .30902 Mdot
.41544 .30902 Mdot
.41561 .30902 Mdot
.41577 .30902 Mdot
.41593 .45617 Mdot
.41611 .30902 Mdot
.41629 .30902 Mdot
.41648 .30902 Mdot
.41667 .16187 Mdot
.41684 .30902 Mdot
.41701 .30902 Mdot
.41718 .45617 Mdot
.41734 .30902 Mdot
.4175 .30902 Mdot
.41768 .45617 Mdot
.41786 .16187 Mdot
.41804 .45617 Mdot
.41824 .30902 Mdot
.41842 .30902 Mdot
.41859 .30902 Mdot
.41875 .30902 Mdot
.41891 .30902 Mdot
.41908 .45617 Mdot
.41925 .16187 Mdot
.41943 .30902 Mdot
.41961 .30902 Mdot
.41982 .30902 Mdot
.41999 .60332 Mdot
.42017 .30902 Mdot
.42033 .45617 Mdot
.42049 .30902 Mdot
.42066 .30902 Mdot
.42083 .45617 Mdot
.42101 .16187 Mdot
.42119 .60332 Mdot
.4214 .30902 Mdot
.42158 .16187 Mdot
.42175 .16187 Mdot
.42191 .30902 Mdot
.42207 .30902 Mdot
.42224 .45617 Mdot
.42241 .30902 Mdot
.42259 .30902 Mdot
.42277 .16187 Mdot
.42297 .30902 Mdot
.42315 .30902 Mdot
.42333 .30902 Mdot
.42349 .16187 Mdot
.42365 .30902 Mdot
.42382 .45617 Mdot
.42399 .30902 Mdot
.42417 .30902 Mdot
.42435 .45617 Mdot
.42455 .30902 Mdot
.42473 .30902 Mdot
.4249 .30902 Mdot
.42507 .45617 Mdot
.42523 .30902 Mdot
.42539 .45617 Mdot
.42556 .30902 Mdot
.42574 .30902 Mdot
.42592 .30902 Mdot
.42612 .45617 Mdot
.4263 .30902 Mdot
.42647 .30902 Mdot
.42664 .16187 Mdot
.4268 .30902 Mdot
.42696 .45617 Mdot
.42713 .30902 Mdot
.42731 .30902 Mdot
.42749 .30902 Mdot
.42768 .16187 Mdot
.42787 .30902 Mdot
.42804 .30902 Mdot
.42821 .30902 Mdot
.42838 .30902 Mdot
.42853 .01472 Mdot
.4287 .45617 Mdot
.42888 .30902 Mdot
.42906 .45617 Mdot
.42925 .16187 Mdot
.42944 .30902 Mdot
.42961 .16187 Mdot
.42978 .30902 Mdot
.42995 .30902 Mdot
.43011 .30902 Mdot
.43027 .30902 Mdot
.43045 .16187 Mdot
.43062 .30902 Mdot
.43081 .30902 Mdot
.43101 .16187 Mdot
.43118 .30902 Mdot
.43135 .30902 Mdot
.43152 .30902 Mdot
.43168 .30902 Mdot
.43184 .30902 Mdot
.43202 .45617 Mdot
.4322 .16187 Mdot
.43238 .30902 Mdot
.43258 .16187 Mdot
.43276 .30902 Mdot
.43293 .45617 Mdot
.4331 .30902 Mdot
.43326 .30902 Mdot
.43342 .45617 Mdot
.4336 .30902 Mdot
.43378 .16187 Mdot
.43396 .30902 Mdot
.43416 .45617 Mdot
.43434 .30902 Mdot
.43451 .30902 Mdot
.43468 .30902 Mdot
.43484 .30902 Mdot
.435 .16187 Mdot
.43518 .30902 Mdot
.43536 .45617 Mdot
.43554 .16187 Mdot
.43574 .16187 Mdot
.43592 .01472 Mdot
.43609 .45617 Mdot
.43626 .30902 Mdot
.43642 .30902 Mdot
.43658 .30902 Mdot
.43676 .30902 Mdot
.43694 .30902 Mdot
.43712 .30902 Mdot
.43732 .30902 Mdot
.4375 .60332 Mdot
.43767 .30902 Mdot
.43784 .30902 Mdot
.438 .30902 Mdot
.43816 .16187 Mdot
.43833 .45617 Mdot
.43851 .30902 Mdot
.43869 .30902 Mdot
.43889 .45617 Mdot
.43907 .30902 Mdot
.43924 .45617 Mdot
.43941 .30902 Mdot
.43957 .30902 Mdot
.43973 .30902 Mdot
.4399 .30902 Mdot
.44008 .16187 Mdot
.44026 .30902 Mdot
.44046 .45617 Mdot
.44064 .16187 Mdot
.44081 .30902 Mdot
.44098 .30902 Mdot
.44115 .30902 Mdot
.4413 .30902 Mdot
.44147 .30902 Mdot
.44165 .45617 Mdot
.44183 .16187 Mdot
.44202 .30902 Mdot
.44221 .30902 Mdot
.44238 .16187 Mdot
.44255 .30902 Mdot
.44272 .01472 Mdot
.44287 .45617 Mdot
.44304 .30902 Mdot
.44322 .30902 Mdot
.44339 .16187 Mdot
.44358 .45617 Mdot
.44377 .16187 Mdot
.44395 .30902 Mdot
.44412 .60332 Mdot
.44429 .30902 Mdot
.44445 .30902 Mdot
.44461 .30902 Mdot
.44479 .16187 Mdot
.44497 .45617 Mdot
.44515 .16187 Mdot
.44535 .16187 Mdot
.44553 .30902 Mdot
.4457 .30902 Mdot
.44587 .30902 Mdot
.44602 .45617 Mdot
.44619 .30902 Mdot
.44637 .30902 Mdot
.44654 .16187 Mdot
.44673 .45617 Mdot
.44693 .16187 Mdot
.44711 .30902 Mdot
.44728 .30902 Mdot
.44745 .16187 Mdot
.44761 .30902 Mdot
.44777 .30902 Mdot
.44795 .30902 Mdot
.44812 .30902 Mdot
.44831 .45617 Mdot
.44851 .30902 Mdot
.44869 .45617 Mdot
.44886 .16187 Mdot
.44903 .30902 Mdot
.44919 .45617 Mdot
.44935 .30902 Mdot
.44952 .30902 Mdot
.4497 .30902 Mdot
.44988 .30902 Mdot
.45008 .30902 Mdot
.45026 .16187 Mdot
.45043 .30902 Mdot
.4506 .30902 Mdot
.45076 .30902 Mdot
.45092 .30902 Mdot
.4511 .30902 Mdot
.45128 .30902 Mdot
.45145 .30902 Mdot
.45165 .30902 Mdot
.45184 .30902 Mdot
.45201 .30902 Mdot
.45218 .30902 Mdot
.45234 .45617 Mdot
.4525 .30902 Mdot
.45267 .30902 Mdot
.45285 .16187 Mdot
.45302 .30902 Mdot
.45322 .30902 Mdot
.45341 .30902 Mdot
.45358 .30902 Mdot
.45375 .30902 Mdot
.45391 .30902 Mdot
.45407 .30902 Mdot
.45424 .30902 Mdot
.45442 .45617 Mdot
.45459 .30902 Mdot
.45479 .45617 Mdot
.45497 .30902 Mdot
.45515 .30902 Mdot
.45532 .30902 Mdot
.45548 .30902 Mdot
.45564 .30902 Mdot
.45581 .30902 Mdot
.45599 .30902 Mdot
.45616 .16187 Mdot
.45635 .16187 Mdot
.45654 .45617 Mdot
.45672 .30902 Mdot
.45689 .30902 Mdot
.45706 .60332 Mdot
.45721 .30902 Mdot
.45738 .30902 Mdot
.45756 .30902 Mdot
.45773 .30902 Mdot
.45792 .45617 Mdot
.45811 .45617 Mdot
.45829 .01472 Mdot
.45846 .30902 Mdot
.45863 .30902 Mdot
.45879 .30902 Mdot
.45895 .30902 Mdot
.45913 .30902 Mdot
.45931 .30902 Mdot
.45949 .30902 Mdot
.45969 .45617 Mdot
.45987 .30902 Mdot
.46004 .30902 Mdot
.46021 .30902 Mdot
.46037 .30902 Mdot
.46053 .16187 Mdot
.46071 .45617 Mdot
.46089 .30902 Mdot
.46107 .30902 Mdot
.46127 .16187 Mdot
.46145 .30902 Mdot
.46162 .30902 Mdot
.46179 .30902 Mdot
.46195 .45617 Mdot
.46211 .45617 Mdot
.46229 .30902 Mdot
.46247 .30902 Mdot
.46265 .30902 Mdot
.46285 .45617 Mdot
.46303 .30902 Mdot
.4632 .30902 Mdot
.46337 .01472 Mdot
.46353 .30902 Mdot
.46369 .16187 Mdot
.46386 .30902 Mdot
.46404 .30902 Mdot
.46422 .30902 Mdot
.46442 .45617 Mdot
.4646 .30902 Mdot
.46478 .30902 Mdot
.46494 .30902 Mdot
.46511 .60332 Mdot
.46527 .16187 Mdot
.46544 .45617 Mdot
.46562 .16187 Mdot
.46579 .45617 Mdot
.46599 .30902 Mdot
.46617 .30902 Mdot
.46635 .30902 Mdot
.46651 .45617 Mdot
.46668 .30902 Mdot
.46684 .30902 Mdot
.46701 .30902 Mdot
.46719 .16187 Mdot
.46736 .30902 Mdot
.46756 .30902 Mdot
.46774 .16187 Mdot
.46792 .30902 Mdot
.46808 .30902 Mdot
.46825 .16187 Mdot
.46841 .30902 Mdot
.46858 .30902 Mdot
.46875 .45617 Mdot
.46893 .30902 Mdot
.46912 .45617 Mdot
.46931 .30902 Mdot
.46948 .30902 Mdot
.46965 .30902 Mdot
.46982 .30902 Mdot
.46998 .30902 Mdot
.47015 .45617 Mdot
.47032 .30902 Mdot
.4705 .30902 Mdot
.47068 .16187 Mdot
.47088 .30902 Mdot
.47106 .30902 Mdot
.47123 .30902 Mdot
.4714 .30902 Mdot
.47155 .30902 Mdot
.47172 .45617 Mdot
.4719 .30902 Mdot
.47207 .45617 Mdot
.47226 .45617 Mdot
.47246 .30902 Mdot
.47264 .30902 Mdot
.47281 .45617 Mdot
.47298 .30902 Mdot
.47314 .30902 Mdot
.4733 .45617 Mdot
.47348 .16187 Mdot
.47365 .30902 Mdot
.47384 .30902 Mdot
.47404 .30902 Mdot
.47422 .30902 Mdot
.47439 .45617 Mdot
.47456 .30902 Mdot
.47472 .30902 Mdot
.47488 .30902 Mdot
.47505 .30902 Mdot
.47523 .30902 Mdot
.47541 .30902 Mdot
.47562 .30902 Mdot
.4758 .30902 Mdot
.47597 .45617 Mdot
.47613 .30902 Mdot
.4763 .30902 Mdot
.47646 .30902 Mdot
.47663 .30902 Mdot
.47681 .45617 Mdot
.47699 .16187 Mdot
.47719 .30902 Mdot
.47737 .45617 Mdot
.47754 .30902 Mdot
.47771 .30902 Mdot
.47787 .30902 Mdot
.47803 .30902 Mdot
.4782 .30902 Mdot
.47838 .30902 Mdot
.47856 .30902 Mdot
.47876 .16187 Mdot
.47894 .60332 Mdot
.47911 .30902 Mdot
.47928 .30902 Mdot
.47945 .30902 Mdot
.47961 .45617 Mdot
.47977 .30902 Mdot
.47995 .30902 Mdot
.48013 .30902 Mdot
.48033 .30902 Mdot
.48051 .45617 Mdot
.48068 .30902 Mdot
.48085 .60332 Mdot
.48102 .30902 Mdot
.48118 .30902 Mdot
.48134 .30902 Mdot
.48152 .30902 Mdot
.4817 .30902 Mdot
.48189 .45617 Mdot
.48208 .16187 Mdot
.48225 .30902 Mdot
.48242 .30902 Mdot
.48259 .30902 Mdot
.48275 .01472 Mdot
.48291 .30902 Mdot
.48309 .30902 Mdot
.48327 .30902 Mdot
.48345 .45617 Mdot
.48365 .16187 Mdot
.48382 .30902 Mdot
.48399 .30902 Mdot
.48416 .16187 Mdot
.48432 .30902 Mdot
.48449 .30902 Mdot
.48466 .30902 Mdot
.48484 .45617 Mdot
.48502 .30902 Mdot
.48522 .30902 Mdot
.4854 .45617 Mdot
.48557 .30902 Mdot
.48574 .30902 Mdot
.4859 .30902 Mdot
.48607 .30902 Mdot
.48624 .45617 Mdot
.48642 .30902 Mdot
.4866 .30902 Mdot
.4868 .30902 Mdot
.48698 .30902 Mdot
.48715 .30902 Mdot
.48732 .30902 Mdot
.48748 .30902 Mdot
.48765 .30902 Mdot
.48782 .16187 Mdot
.488 .45617 Mdot
.48818 .16187 Mdot
.48838 .30902 Mdot
.48856 .45617 Mdot
.48873 .30902 Mdot
.4889 .30902 Mdot
.48906 .16187 Mdot
.48922 .30902 Mdot
.4894 .45617 Mdot
.48958 .30902 Mdot
.48976 .30902 Mdot
.48996 .16187 Mdot
.49014 .30902 Mdot
.49031 .30902 Mdot
.49048 .30902 Mdot
.49064 .30902 Mdot
.4908 .45617 Mdot
.49097 .01472 Mdot
.49115 .45617 Mdot
.49133 .45617 Mdot
.49153 .16187 Mdot
.49171 .30902 Mdot
.49188 .30902 Mdot
.49205 .45617 Mdot
.49221 .30902 Mdot
.49237 .16187 Mdot
.49254 .30902 Mdot
.49272 .16187 Mdot
.4929 .16187 Mdot
.4931 .45617 Mdot
.49328 .30902 Mdot
.49345 .30902 Mdot
.49362 .30902 Mdot
.49379 .01472 Mdot
.49395 .30902 Mdot
.49411 .30902 Mdot
.49429 .30902 Mdot
.49447 .16187 Mdot
.49466 .45617 Mdot
.49485 .01472 Mdot
.49502 .60332 Mdot
.49519 .30902 Mdot
.49536 .30902 Mdot
.49552 .16187 Mdot
.49568 .30902 Mdot
.49586 .30902 Mdot
.49604 .30902 Mdot
.49622 .30902 Mdot
.49642 .30902 Mdot
.49659 .30902 Mdot
.49676 .30902 Mdot
.49693 .30902 Mdot
.49709 .30902 Mdot
.49725 .30902 Mdot
.49743 .16187 Mdot
.49761 .45617 Mdot
.49779 .30902 Mdot
.49799 .30902 Mdot
.49817 .30902 Mdot
.49834 .45617 Mdot
.49851 .16187 Mdot
.49867 .30902 Mdot
.49883 .45617 Mdot
.49901 .45617 Mdot
.49919 .16187 Mdot
.49937 .45617 Mdot
.49957 .30902 Mdot
.49975 .30902 Mdot
.49992 .45617 Mdot
.50009 .30902 Mdot
.50025 .30902 Mdot
.50041 .30902 Mdot
.50059 .30902 Mdot
.50077 .30902 Mdot
.50095 .30902 Mdot
.50115 .16187 Mdot
.50133 .30902 Mdot
.5015 .30902 Mdot
.50167 .30902 Mdot
.50183 .30902 Mdot
.50199 .30902 Mdot
.50216 .30902 Mdot
.50234 .16187 Mdot
.50252 .45617 Mdot
.50272 .30902 Mdot
.5029 .45617 Mdot
.50308 .30902 Mdot
.50324 .30902 Mdot
.50341 .30902 Mdot
.50357 .30902 Mdot
.50374 .30902 Mdot
.50392 .30902 Mdot
.5041 .30902 Mdot
.5043 .30902 Mdot
.50448 .01472 Mdot
.50465 .30902 Mdot
.50482 .30902 Mdot
.50498 .45617 Mdot
.50514 .16187 Mdot
.50531 .30902 Mdot
.50549 .30902 Mdot
.50567 .30902 Mdot
.50587 .30902 Mdot
.50605 .30902 Mdot
.50622 .30902 Mdot
.50639 .30902 Mdot
.50655 .45617 Mdot
.50671 .30902 Mdot
.50688 .16187 Mdot
.50706 .16187 Mdot
.50724 .30902 Mdot
.50743 .30902 Mdot
.50762 .30902 Mdot
.50779 .30902 Mdot
.50796 .30902 Mdot
.50813 .60332 Mdot
.50828 .30902 Mdot
.50845 .30902 Mdot
.50863 .30902 Mdot
.5088 .16187 Mdot
.50899 .30902 Mdot
.50918 .30902 Mdot
.50936 .45617 Mdot
.50953 .30902 Mdot
.5097 .30902 Mdot
.50986 .16187 Mdot
.51002 .60332 Mdot
.5102 .16187 Mdot
.51037 .30902 Mdot
.51056 .30902 Mdot
.51076 .30902 Mdot
.51093 .30902 Mdot
.5111 .30902 Mdot
.51127 .30902 Mdot
.51143 .30902 Mdot
.5116 .30902 Mdot
.51177 .45617 Mdot
.51195 .30902 Mdot
.51213 .16187 Mdot
.51233 .30902 Mdot
.51251 .30902 Mdot
.51268 .30902 Mdot
.51285 .30902 Mdot
.51301 .30902 Mdot
.51318 .30902 Mdot
.51335 .30902 Mdot
.51353 .30902 Mdot
.51371 .45617 Mdot
.51391 .30902 Mdot
.51409 .30902 Mdot
.51427 .30902 Mdot
.51443 .01472 Mdot
.51459 .30902 Mdot
.51476 .45617 Mdot
.51493 .30902 Mdot
.51511 .30902 Mdot
.51529 .30902 Mdot
.51549 .30902 Mdot
.51567 .30902 Mdot
.51584 .30902 Mdot
.51601 .30902 Mdot
.51617 .16187 Mdot
.51633 .30902 Mdot
.51651 .45617 Mdot
.51669 .45617 Mdot
.51686 .45617 Mdot
.51706 .30902 Mdot
.51724 .30902 Mdot
.51742 .30902 Mdot
.51758 .30902 Mdot
.51775 .30902 Mdot
.51791 .30902 Mdot
.51808 .16187 Mdot
.51826 .30902 Mdot
.51843 .30902 Mdot
.51863 .30902 Mdot
.51882 .30902 Mdot
.51899 .60332 Mdot
.51916 .30902 Mdot
.51932 .16187 Mdot
.51948 .30902 Mdot
.51965 .45617 Mdot
.51983 .30902 Mdot
.52 .30902 Mdot
.5202 .45617 Mdot
.52038 .30902 Mdot
.52056 .30902 Mdot
.52073 .30902 Mdot
.52089 .30902 Mdot
.52105 .01472 Mdot
.52122 .30902 Mdot
.52139 .30902 Mdot
.52157 .45617 Mdot
.52176 .16187 Mdot
.52195 .30902 Mdot
.52213 .30902 Mdot
.52229 .30902 Mdot
.52246 .30902 Mdot
.52262 .30902 Mdot
.52279 .30902 Mdot
.52296 .45617 Mdot
.52314 .16187 Mdot
.52333 .30902 Mdot
.52352 .30902 Mdot
.5237 .30902 Mdot
.52387 .30902 Mdot
.52404 .30902 Mdot
.5242 .16187 Mdot
.52436 .30902 Mdot
.52454 .45617 Mdot
.52472 .30902 Mdot
.5249 .30902 Mdot
.5251 .30902 Mdot
.52528 .30902 Mdot
.52545 .30902 Mdot
.52562 .01472 Mdot
.52578 .30902 Mdot
.52594 .30902 Mdot
.52612 .45617 Mdot
.5263 .45617 Mdot
.52648 .30902 Mdot
.52668 .30902 Mdot
.52686 .30902 Mdot
.52703 .30902 Mdot
.5272 .30902 Mdot
.52736 .30902 Mdot
.52752 .45617 Mdot
.5277 .16187 Mdot
.52788 .30902 Mdot
.52806 .45617 Mdot
.52826 .16187 Mdot
.52844 .30902 Mdot
.52861 .30902 Mdot
.52878 .30902 Mdot
.52894 .30902 Mdot
.5291 .16187 Mdot
.52927 .45617 Mdot
.52945 .30902 Mdot
.52963 .16187 Mdot
.52983 .30902 Mdot
.53001 .30902 Mdot
.53018 .30902 Mdot
.53035 .45617 Mdot
.53051 .30902 Mdot
.53067 .60332 Mdot
.53085 .16187 Mdot
.53102 .16187 Mdot
.5312 .30902 Mdot
.5314 .45617 Mdot
.53158 .16187 Mdot
.53176 .30902 Mdot
.53192 .30902 Mdot
.53209 .30902 Mdot
.53225 .45617 Mdot
.53242 .30902 Mdot
.53259 .30902 Mdot
.53277 .16187 Mdot
.53297 .45617 Mdot
.53315 .30902 Mdot
.53333 .30902 Mdot
.53349 .30902 Mdot
.53366 .30902 Mdot
.53382 .16187 Mdot
.53399 .30902 Mdot
.53416 .30902 Mdot
.53434 .30902 Mdot
.53453 .30902 Mdot
.53472 .16187 Mdot
.53489 .30902 Mdot
.53506 .30902 Mdot
.53523 .30902 Mdot
.53539 .30902 Mdot
.53555 .30902 Mdot
.53573 .30902 Mdot
.53591 .30902 Mdot
.53609 .45617 Mdot
.53629 .30902 Mdot
.53647 .30902 Mdot
.53663 .30902 Mdot
.5368 .30902 Mdot
.53696 .30902 Mdot
.53713 .30902 Mdot
.5373 .30902 Mdot
.53748 .45617 Mdot
.53766 .30902 Mdot
.53787 .30902 Mdot
.53804 .16187 Mdot
.53822 .30902 Mdot
.53838 .60332 Mdot
.53854 .01472 Mdot
.53871 .30902 Mdot
.53888 .30902 Mdot
.53906 .45617 Mdot
.53924 .30902 Mdot
.53945 .30902 Mdot
.53963 .16187 Mdot
.5398 .45617 Mdot
.53996 .30902 Mdot
.54012 .30902 Mdot
.54029 .30902 Mdot
.54046 .30902 Mdot
.54064 .16187 Mdot
.54082 .30902 Mdot
.54102 .30902 Mdot
.5412 .30902 Mdot
.54138 .30902 Mdot
.54154 .45617 Mdot
.5417 .16187 Mdot
.54187 .30902 Mdot
.54204 .30902 Mdot
.54222 .30902 Mdot
.5424 .45617 Mdot
.5426 .45617 Mdot
.54278 .30902 Mdot
.54295 .45617 Mdot
.54312 .30902 Mdot
.54328 .30902 Mdot
.54344 .30902 Mdot
.54361 .30902 Mdot
.54379 .30902 Mdot
.54397 .45617 Mdot
.54417 .45617 Mdot
.54435 .30902 Mdot
.54453 .30902 Mdot
.54469 .30902 Mdot
.54486 .30902 Mdot
.54502 .30902 Mdot
.54519 .16187 Mdot
.54536 .16187 Mdot
.54554 .16187 Mdot
.54574 .30902 Mdot
.54592 .16187 Mdot
.5461 .30902 Mdot
.54626 .30902 Mdot
.54643 .30902 Mdot
.54659 .30902 Mdot
.54676 .30902 Mdot
.54693 .45617 Mdot
.54711 .30902 Mdot
.5473 .30902 Mdot
.54749 .60332 Mdot
.54766 .30902 Mdot
.54783 .16187 Mdot
.548 .30902 Mdot
.54816 .30902 Mdot
.54832 .45617 Mdot
.5485 .30902 Mdot
.54868 .30902 Mdot
.54886 .16187 Mdot
.54906 .30902 Mdot
.54923 .16187 Mdot
.5494 .16187 Mdot
.54957 .30902 Mdot
.54973 .01472 Mdot
.5499 .30902 Mdot
.55007 .30902 Mdot
.55025 .30902 Mdot
.55043 .45617 Mdot
.55063 .30902 Mdot
.55081 .30902 Mdot
.55098 .30902 Mdot
.55115 .30902 Mdot
.55131 .30902 Mdot
.55147 .30902 Mdot
.55165 .30902 Mdot
.55183 .30902 Mdot
.55201 .30902 Mdot
.55221 .30902 Mdot
.55239 .30902 Mdot
.55256 .01472 Mdot
.55273 .30902 Mdot
.55289 .16187 Mdot
.55305 .30902 Mdot
.55323 .16187 Mdot
.55341 .45617 Mdot
.55359 .45617 Mdot
.55379 .30902 Mdot
.55397 .30902 Mdot
.55414 .30902 Mdot
.55431 .30902 Mdot
.55447 .30902 Mdot
.55463 .45617 Mdot
.55481 .30902 Mdot
.55499 .16187 Mdot
.55517 .30902 Mdot
.55537 .30902 Mdot
.55555 .30902 Mdot
.55572 .30902 Mdot
.55589 .30902 Mdot
.55605 .30902 Mdot
.55621 .16187 Mdot
.55638 .30902 Mdot
.55656 .30902 Mdot
.55674 .16187 Mdot
.55694 .45617 Mdot
.55712 .30902 Mdot
.55729 .30902 Mdot
.55746 .30902 Mdot
.55762 .30902 Mdot
.55778 .30902 Mdot
.55795 .45617 Mdot
.55813 .30902 Mdot
.55831 .16187 Mdot
.55851 .30902 Mdot
.55869 .30902 Mdot
.55886 .30902 Mdot
.55903 .16187 Mdot
.5592 .30902 Mdot
.55936 .30902 Mdot
.55952 .30902 Mdot
.5597 .30902 Mdot
.55988 .30902 Mdot
.56007 .30902 Mdot
.56026 .30902 Mdot
.56043 .30902 Mdot
.5606 .30902 Mdot
.56077 .30902 Mdot
.56093 .60332 Mdot
.56109 .30902 Mdot
.56127 .30902 Mdot
.56145 .01472 Mdot
.56163 .45617 Mdot
.56183 .30902 Mdot
.562 .30902 Mdot
.56217 .30902 Mdot
.56234 .30902 Mdot
.5625 .16187 Mdot
.56266 .45617 Mdot
.56284 .30902 Mdot
.56302 .16187 Mdot
.5632 .30902 Mdot
.5634 .30902 Mdot
.56358 .30902 Mdot
.56375 .30902 Mdot
.56392 .30902 Mdot
.56407 .16187 Mdot
.56424 .45617 Mdot
.56441 .30902 Mdot
.56459 .30902 Mdot
.56478 .45617 Mdot
.56498 .16187 Mdot
.56516 .30902 Mdot
.56533 .30902 Mdot
.5655 .30902 Mdot
.56566 .60332 Mdot
.56582 .30902 Mdot
.566 .45617 Mdot
.56617 .30902 Mdot
.56636 .30902 Mdot
.56656 .16187 Mdot
.56674 .01472 Mdot
.56691 .16187 Mdot
.56708 .45617 Mdot
.56724 .30902 Mdot
.5674 .30902 Mdot
.56757 .30902 Mdot
.56775 .30902 Mdot
.56793 .45617 Mdot
.56813 .30902 Mdot
.56831 .30902 Mdot
.56849 .30902 Mdot
.56865 .30902 Mdot
.56881 .30902 Mdot
.56898 .30902 Mdot
.56915 .16187 Mdot
.56933 .45617 Mdot
.56951 .16187 Mdot
.56971 .30902 Mdot
.56989 .16187 Mdot
.57006 .30902 Mdot
.57023 .16187 Mdot
.57039 .30902 Mdot
.57055 .30902 Mdot
.57072 .30902 Mdot
.5709 .30902 Mdot
.57108 .30902 Mdot
.57128 .30902 Mdot
.57146 .30902 Mdot
.57163 .60332 Mdot
.5718 .30902 Mdot
.57196 .16187 Mdot
.57212 .30902 Mdot
.57229 .16187 Mdot
.57247 .30902 Mdot
.57265 .30902 Mdot
.57284 .16187 Mdot
.57303 .45617 Mdot
.5732 .30902 Mdot
.57337 .30902 Mdot
.57353 .30902 Mdot
.57369 .30902 Mdot
.57386 .16187 Mdot
.57404 .30902 Mdot
.57421 .16187 Mdot
.5744 .30902 Mdot
.57459 .45617 Mdot
.57477 .30902 Mdot
.57494 .16187 Mdot
.57511 .30902 Mdot
.57526 .30902 Mdot
.57543 .16187 Mdot
.57561 .30902 Mdot
.57578 .30902 Mdot
.57597 .30902 Mdot
.57617 .30902 Mdot
.57634 .30902 Mdot
.57651 .30902 Mdot
.57668 .30902 Mdot
.57684 .30902 Mdot
.577 .30902 Mdot
.57718 .30902 Mdot
.57736 .45617 Mdot
.57754 .30902 Mdot
.57774 .30902 Mdot
.57792 .30902 Mdot
.57809 .30902 Mdot
.57826 .30902 Mdot
.57842 .30902 Mdot
.57858 .16187 Mdot
.57876 .30902 Mdot
.57894 .30902 Mdot
.57912 .30902 Mdot
.57932 .30902 Mdot
.5795 .45617 Mdot
.57967 .30902 Mdot
.57984 .30902 Mdot
.58 .16187 Mdot
.58016 .45617 Mdot
.58034 .30902 Mdot
.58052 .30902 Mdot
.5807 .16187 Mdot
.5809 .45617 Mdot
.58108 .45617 Mdot
.58125 .16187 Mdot
.58142 .30902 Mdot
.58158 .30902 Mdot
.58174 .30902 Mdot
.58191 .16187 Mdot
.5821 .30902 Mdot
.58227 .30902 Mdot
.58247 .30902 Mdot
.58265 .30902 Mdot
.58283 .16187 Mdot
.58299 .30902 Mdot
.58316 .30902 Mdot
.58332 .30902 Mdot
.58349 .45617 Mdot
.58367 .45617 Mdot
.58384 .45617 Mdot
.58404 .30902 Mdot
.58422 .30902 Mdot
.5844 .30902 Mdot
.58457 .30902 Mdot
.58473 .30902 Mdot
.58489 .30902 Mdot
.58506 .45617 Mdot
.58524 .30902 Mdot
.58541 .45617 Mdot
.58561 .16187 Mdot
.58579 .30902 Mdot
.58597 .30902 Mdot
.58614 .45617 Mdot
.5863 .30902 Mdot
.58646 .30902 Mdot
.58663 .16187 Mdot
.5868 .30902 Mdot
.58698 .30902 Mdot
.58717 .30902 Mdot
.58736 .45617 Mdot
.58754 .30902 Mdot
.5877 .45617 Mdot
.58787 .30902 Mdot
.58803 .30902 Mdot
.5882 .45617 Mdot
.58837 .16187 Mdot
.58855 .45617 Mdot
.58874 .01472 Mdot
.58893 .16187 Mdot
.58911 .16187 Mdot
.58928 .30902 Mdot
.58945 .30902 Mdot
.5896 .30902 Mdot
.58977 .30902 Mdot
.58995 .30902 Mdot
.59012 .30902 Mdot
.59031 .30902 Mdot
.59051 .30902 Mdot
.59069 .45617 Mdot
.59086 .30902 Mdot
.59103 .30902 Mdot
.59119 .01472 Mdot
.59135 .45617 Mdot
.59153 .16187 Mdot
.5917 .30902 Mdot
.59189 .30902 Mdot
.59209 .16187 Mdot
.59227 .01472 Mdot
.59244 .30902 Mdot
.59261 .30902 Mdot
.59277 .30902 Mdot
.59293 .30902 Mdot
.59311 .30902 Mdot
.59329 .30902 Mdot
.59347 .30902 Mdot
.59367 .60332 Mdot
.59385 .30902 Mdot
.59402 .30902 Mdot
.59419 .30902 Mdot
.59435 .30902 Mdot
.59451 .45617 Mdot
.59468 .16187 Mdot
.59486 .30902 Mdot
.59504 .30902 Mdot
.59524 .30902 Mdot
.59542 .30902 Mdot
.59559 .30902 Mdot
.59576 .30902 Mdot
.59592 .30902 Mdot
.59608 .30902 Mdot
.59626 .30902 Mdot
.59644 .45617 Mdot
.59661 .16187 Mdot
.59681 .30902 Mdot
.59699 .30902 Mdot
.59717 .30902 Mdot
.59733 .45617 Mdot
.5975 .30902 Mdot
.59766 .16187 Mdot
.59783 .30902 Mdot
.59801 .16187 Mdot
.59818 .30902 Mdot
.59838 .30902 Mdot
.59856 .30902 Mdot
.59874 .30902 Mdot
.5989 .30902 Mdot
.59907 .16187 Mdot
.59923 .30902 Mdot
.5994 .30902 Mdot
.59957 .30902 Mdot
.59975 .30902 Mdot
.59994 .30902 Mdot
.60013 .01472 Mdot
.60031 .30902 Mdot
.60047 .30902 Mdot
.60064 .16187 Mdot
.6008 .30902 Mdot
.60097 .30902 Mdot
.60114 .16187 Mdot
.60132 .60332 Mdot
.60151 .30902 Mdot
.6017 .30902 Mdot
.60188 .30902 Mdot
.60204 .16187 Mdot
.60221 .30902 Mdot
.60237 .30902 Mdot
.60254 .30902 Mdot
.60271 .30902 Mdot
.60289 .30902 Mdot
.60307 .30902 Mdot
.60327 .45617 Mdot
.60345 .30902 Mdot
.60362 .30902 Mdot
.60379 .30902 Mdot
.60395 .30902 Mdot
.60412 .30902 Mdot
.60429 .30902 Mdot
.60447 .16187 Mdot
.60465 .30902 Mdot
.60485 .30902 Mdot
.60503 .16187 Mdot
.6052 .30902 Mdot
.60537 .30902 Mdot
.60553 .30902 Mdot
.6057 .16187 Mdot
.60587 .30902 Mdot
.60605 .45617 Mdot
.60623 .16187 Mdot
.60643 .45617 Mdot
.60661 .16187 Mdot
.60678 .30902 Mdot
.60695 .30902 Mdot
.60711 .45617 Mdot
.60727 .30902 Mdot
.60745 .45617 Mdot
.60763 .30902 Mdot
.60781 .30902 Mdot
.60801 .30902 Mdot
.60819 .30902 Mdot
.60836 .30902 Mdot
.60853 .30902 Mdot
.60869 .30902 Mdot
.60885 .30902 Mdot
.60902 .45617 Mdot
.6092 .30902 Mdot
.60938 .30902 Mdot
.60958 .30902 Mdot
.60976 .45617 Mdot
.60993 .30902 Mdot
.6101 .30902 Mdot
.61026 .30902 Mdot
.61042 .30902 Mdot
.61059 .30902 Mdot
.61077 .45617 Mdot
.61095 .30902 Mdot
.61115 .45617 Mdot
.61133 .01472 Mdot
.61151 .01472 Mdot
.61167 .30902 Mdot
.61184 .30902 Mdot
.612 .30902 Mdot
.61217 .16187 Mdot
.61234 .30902 Mdot
.61252 .16187 Mdot
.61271 .30902 Mdot
.6129 .30902 Mdot
.61307 .30902 Mdot
.61324 .30902 Mdot
.61341 .30902 Mdot
.61357 .16187 Mdot
.61373 .30902 Mdot
.61391 .30902 Mdot
.61409 .30902 Mdot
.61428 .30902 Mdot
.61447 .45617 Mdot
.61464 .30902 Mdot
.61481 .30902 Mdot
.61498 .30902 Mdot
.61514 .16187 Mdot
.6153 .30902 Mdot
.61548 .30902 Mdot
.61566 .45617 Mdot
.61584 .30902 Mdot
.61604 .30902 Mdot
.61622 .45617 Mdot
.61639 .30902 Mdot
.61656 .45617 Mdot
.61672 .30902 Mdot
.61688 .30902 Mdot
.61706 .30902 Mdot
.61723 .30902 Mdot
.61742 .16187 Mdot
.61762 .30902 Mdot
.6178 .30902 Mdot
.61797 .30902 Mdot
.61814 .30902 Mdot
.6183 .30902 Mdot
.61846 .45617 Mdot
.61864 .45617 Mdot
.61882 .30902 Mdot
.619 .30902 Mdot
.6192 .30902 Mdot
.61938 .01472 Mdot
.61955 .30902 Mdot
.61972 .30902 Mdot
.61988 .30902 Mdot
.62004 .45617 Mdot
.62021 .30902 Mdot
.6204 .16187 Mdot
.62057 .30902 Mdot
.62078 .16187 Mdot
.62096 .30902 Mdot
.62113 .60332 Mdot
.62129 .30902 Mdot
.62146 .30902 Mdot
.62162 .30902 Mdot
.62179 .16187 Mdot
.62197 .30902 Mdot
.62215 .16187 Mdot
.62235 .30902 Mdot
.62253 .16187 Mdot
.6227 .30902 Mdot
.62287 .30902 Mdot
.62303 .30902 Mdot
.62319 .30902 Mdot
.62336 .30902 Mdot
.62354 .30902 Mdot
.62372 .30902 Mdot
.62392 .30902 Mdot
.6241 .01472 Mdot
.62427 .16187 Mdot
.62444 .30902 Mdot
.62461 .30902 Mdot
.62476 .30902 Mdot
.62493 .45617 Mdot
.62511 .16187 Mdot
.62529 .45617 Mdot
.62548 .30902 Mdot
.62567 .30902 Mdot
.62584 .30902 Mdot
.62601 .30902 Mdot
.62618 .30902 Mdot
.62633 .30902 Mdot
.6265 .45617 Mdot
.62668 .30902 Mdot
.62685 .16187 Mdot
.62704 .30902 Mdot
.62723 .30902 Mdot
.62741 .45617 Mdot
.62758 .30902 Mdot
.62775 .30902 Mdot
.62791 .30902 Mdot
.62807 .30902 Mdot
.62825 .30902 Mdot
.62842 .30902 Mdot
.62861 .45617 Mdot
.62881 .16187 Mdot
.62898 .45617 Mdot
.62915 .30902 Mdot
.62932 .16187 Mdot
.62948 .30902 Mdot
.62965 .30902 Mdot
.62982 .45617 Mdot
.63 .30902 Mdot
.63018 .30902 Mdot
.63038 .45617 Mdot
.63056 .30902 Mdot
.63073 .30902 Mdot
.6309 .30902 Mdot
.63106 .30902 Mdot
.63123 .30902 Mdot
.6314 .16187 Mdot
.63158 .60332 Mdot
.63176 .30902 Mdot
.63196 .30902 Mdot
.63214 .30902 Mdot
.63231 .30902 Mdot
.63248 .30902 Mdot
.63264 .30902 Mdot
.63281 .30902 Mdot
.63298 .30902 Mdot
.63316 .30902 Mdot
.63334 .45617 Mdot
.63354 .45617 Mdot
.63372 .30902 Mdot
.63389 .30902 Mdot
.63406 .16187 Mdot
.63422 .30902 Mdot
.63438 .30902 Mdot
.63456 .30902 Mdot
.63474 .45617 Mdot
.63491 .30902 Mdot
.63511 .16187 Mdot
.6353 .30902 Mdot
.63547 .30902 Mdot
.63564 .30902 Mdot
.6358 .30902 Mdot
.63596 .30902 Mdot
.63613 .45617 Mdot
.63631 .30902 Mdot
.63649 .30902 Mdot
.63668 .16187 Mdot
.63687 .60332 Mdot
.63704 .30902 Mdot
.63721 .30902 Mdot
.63737 .30902 Mdot
.63753 .30902 Mdot
.6377 .45617 Mdot
.63788 .30902 Mdot
.63805 .30902 Mdot
.63825 .30902 Mdot
.63843 .30902 Mdot
.63861 .30902 Mdot
.63878 .30902 Mdot
.63894 .30902 Mdot
.6391 .30902 Mdot
.63927 .30902 Mdot
.63945 .45617 Mdot
.63962 .30902 Mdot
.63981 .30902 Mdot
.64 .30902 Mdot
.64018 .45617 Mdot
.64035 .16187 Mdot
.64051 .01472 Mdot
.64067 .45617 Mdot
.64084 .30902 Mdot
.64101 .16187 Mdot
.64119 .30902 Mdot
.64138 .30902 Mdot
.64157 .16187 Mdot
.64175 .45617 Mdot
.64192 .30902 Mdot
.64209 .30902 Mdot
.64225 .30902 Mdot
.64241 .30902 Mdot
.64259 .30902 Mdot
.64276 .30902 Mdot
.64295 .30902 Mdot
.64315 .30902 Mdot
.64333 .30902 Mdot
.6435 .30902 Mdot
.64367 .30902 Mdot
.64383 .30902 Mdot
.64399 .30902 Mdot
.64417 .45617 Mdot
.64435 .30902 Mdot
.64453 .30902 Mdot
.64473 .16187 Mdot
.64491 .60332 Mdot
.64508 .01472 Mdot
.64525 .30902 Mdot
.64541 .30902 Mdot
.64557 .01472 Mdot
.64575 .45617 Mdot
.64593 .30902 Mdot
.64611 .45617 Mdot
.64631 .30902 Mdot
.64649 .16187 Mdot
.64666 .16187 Mdot
.64683 .30902 Mdot
.64699 .30902 Mdot
.64715 .30902 Mdot
.64732 .30902 Mdot
.6475 .30902 Mdot
.64768 .30902 Mdot
.64788 .30902 Mdot
.64806 .30902 Mdot
.64824 .30902 Mdot
.6484 .30902 Mdot
.64856 .30902 Mdot
.64873 .30902 Mdot
.6489 .30902 Mdot
.64908 .30902 Mdot
.64925 .30902 Mdot
.64945 .30902 Mdot
.64963 .60332 Mdot
.64981 .30902 Mdot
.64998 .30902 Mdot
.65014 .30902 Mdot
.6503 .16187 Mdot
.65047 .30902 Mdot
.65065 .30902 Mdot
.65082 .30902 Mdot
.65102 .30902 Mdot
.6512 .16187 Mdot
.65138 .60332 Mdot
.65155 .30902 Mdot
.65171 .30902 Mdot
.65187 .30902 Mdot
.65204 .30902 Mdot
.65222 .45617 Mdot
.65239 .30902 Mdot
.65258 .30902 Mdot
.65277 .30902 Mdot
.65295 .30902 Mdot
.65311 .30902 Mdot
.65328 .45617 Mdot
.65344 .30902 Mdot
.65361 .60332 Mdot
.65378 .30902 Mdot
.65396 .30902 Mdot
.65415 .30902 Mdot
.65434 .30902 Mdot
.65452 .30902 Mdot
.65469 .16187 Mdot
.65486 .30902 Mdot
.65501 .30902 Mdot
.65518 .16187 Mdot
.65535 .30902 Mdot
.65553 .45617 Mdot
.65572 .30902 Mdot
.65592 .30902 Mdot
.65609 .30902 Mdot
.65626 .30902 Mdot
.65643 .30902 Mdot
.65659 .30902 Mdot
.65676 .30902 Mdot
.65693 .30902 Mdot
.65711 .30902 Mdot
.65729 .30902 Mdot
.6575 .30902 Mdot
.65767 .16187 Mdot
.65785 .30902 Mdot
.65801 .16187 Mdot
.65817 .30902 Mdot
.65834 .30902 Mdot
.65851 .30902 Mdot
.65869 .30902 Mdot
.65887 .16187 Mdot
.65907 .16187 Mdot
.65925 .30902 Mdot
.65943 .45617 Mdot
.65959 .30902 Mdot
.65975 .01472 Mdot
.65992 .30902 Mdot
.66009 .30902 Mdot
.66027 .16187 Mdot
.66045 .45617 Mdot
.66065 .30902 Mdot
.66083 .30902 Mdot
.661 .30902 Mdot
.66117 .30902 Mdot
.66133 .16187 Mdot
.66149 .30902 Mdot
.66166 .30902 Mdot
.66184 .16187 Mdot
.66202 .30902 Mdot
.66222 .30902 Mdot
.6624 .30902 Mdot
.66258 .30902 Mdot
.66274 .30902 Mdot
.66291 .30902 Mdot
.66307 .30902 Mdot
.66324 .45617 Mdot
.66341 .16187 Mdot
.66359 .30902 Mdot
.66379 .45617 Mdot
.66397 .30902 Mdot
.66415 .30902 Mdot
.66431 .30902 Mdot
.66448 .30902 Mdot
.66464 .16187 Mdot
.66481 .30902 Mdot
.66499 .30902 Mdot
.66516 .45617 Mdot
.66535 .30902 Mdot
.66554 .16187 Mdot
.66572 .30902 Mdot
.66588 .01472 Mdot
.66605 .30902 Mdot
.66621 .30902 Mdot
.66638 .16187 Mdot
.66655 .30902 Mdot
.66673 .30902 Mdot
.66692 .30902 Mdot
.66711 .30902 Mdot
.66729 .30902 Mdot
.66745 .30902 Mdot
.66762 .30902 Mdot
.66778 .45617 Mdot
.66795 .30902 Mdot
.66812 .30902 Mdot
.6683 .30902 Mdot
.66848 .60332 Mdot
.66868 .30902 Mdot
.66886 .30902 Mdot
.66903 .30902 Mdot
.6692 .30902 Mdot
.66936 .30902 Mdot
.66952 .16187 Mdot
.6697 .30902 Mdot
.66988 .30902 Mdot
.67006 .30902 Mdot
.67026 .45617 Mdot
.67044 .30902 Mdot
.67061 .30902 Mdot
.67078 .30902 Mdot
.67094 .01472 Mdot
.6711 .30902 Mdot
.67128 .30902 Mdot
.67146 .45617 Mdot
.67164 .30902 Mdot
.67184 .30902 Mdot
.67202 .30902 Mdot
.67219 .30902 Mdot
.67236 .30902 Mdot
.67252 .30902 Mdot
.67268 .16187 Mdot
.67286 .30902 Mdot
.67304 .30902 Mdot
.67322 .30902 Mdot
.67342 .45617 Mdot
.6736 .30902 Mdot
.67377 .30902 Mdot
.67394 .60332 Mdot
.6741 .30902 Mdot
.67426 .30902 Mdot
.67443 .16187 Mdot
.67461 .30902 Mdot
.67479 .16187 Mdot
.67499 .30902 Mdot
.67517 .30902 Mdot
.67534 .45617 Mdot
.67551 .01472 Mdot
.67567 .30902 Mdot
.67583 .30902 Mdot
.676 .30902 Mdot
.67618 .16187 Mdot
.67636 .16187 Mdot
.67656 .30902 Mdot
.67674 .30902 Mdot
.67691 .30902 Mdot
.67708 .30902 Mdot
.67725 .01472 Mdot
.67741 .30902 Mdot
.67757 .16187 Mdot
.67775 .45617 Mdot
.67793 .30902 Mdot
.67812 .30902 Mdot
.67831 .16187 Mdot
.67848 .30902 Mdot
.67865 .30902 Mdot
.67882 .45617 Mdot
.67898 .30902 Mdot
.67914 .16187 Mdot
.67932 .45617 Mdot
.6795 .30902 Mdot
.67969 .16187 Mdot
.67988 .30902 Mdot
.68005 .16187 Mdot
.68022 .30902 Mdot
.68039 .30902 Mdot
.68055 .30902 Mdot
.68071 .30902 Mdot
.68089 .16187 Mdot
.68107 .30902 Mdot
.68125 .16187 Mdot
.68145 .45617 Mdot
.68163 .30902 Mdot
.6818 .30902 Mdot
.68196 .30902 Mdot
.68212 .45617 Mdot
.68229 .45617 Mdot
.68246 .30902 Mdot
.68264 .45617 Mdot
.68283 .30902 Mdot
.68303 .16187 Mdot
.68321 .45617 Mdot
.68338 .30902 Mdot
.68355 .45617 Mdot
.6837 .45617 Mdot
.68387 .30902 Mdot
.68404 .30902 Mdot
.68422 .30902 Mdot
.68441 .16187 Mdot
.68461 .30902 Mdot
.68479 .30902 Mdot
.68496 .30902 Mdot
.68513 .30902 Mdot
.68528 .30902 Mdot
.68545 .16187 Mdot
.68562 .30902 Mdot
.6858 .45617 Mdot
.68598 .30902 Mdot
.68618 .30902 Mdot
.68636 .30902 Mdot
.68653 .30902 Mdot
.6867 .30902 Mdot
.68686 .30902 Mdot
.68703 .30902 Mdot
.6872 .30902 Mdot
.68738 .30902 Mdot
.68756 .16187 Mdot
.68776 .30902 Mdot
.68794 .45617 Mdot
.68811 .30902 Mdot
.68828 .30902 Mdot
.68844 .30902 Mdot
.6886 .30902 Mdot
.68877 .30902 Mdot
.68895 .30902 Mdot
.68913 .30902 Mdot
.68933 .30902 Mdot
.68951 .45617 Mdot
.68968 .30902 Mdot
.68985 .30902 Mdot
.69001 .30902 Mdot
.69017 .30902 Mdot
.69034 .30902 Mdot
.69052 .45617 Mdot
.6907 .45617 Mdot
.69089 .30902 Mdot
.69108 .30902 Mdot
.69125 .01472 Mdot
.69142 .30902 Mdot
.69158 .30902 Mdot
.69174 .30902 Mdot
.69191 .30902 Mdot
.69209 .16187 Mdot
.69226 .30902 Mdot
.69245 .30902 Mdot
.69264 .45617 Mdot
.69282 .30902 Mdot
.69299 .60332 Mdot
.69316 .16187 Mdot
.69331 .30902 Mdot
.69348 .16187 Mdot
.69366 .45617 Mdot
.69383 .30902 Mdot
.69402 .30902 Mdot
.69421 .30902 Mdot
.69439 .30902 Mdot
.69456 .30902 Mdot
.69473 .30902 Mdot
.69489 .30902 Mdot
.69505 .30902 Mdot
.69523 .16187 Mdot
.69541 .30902 Mdot
.69559 .30902 Mdot
.69579 .16187 Mdot
.69597 .30902 Mdot
.69614 .30902 Mdot
.69631 .45617 Mdot
.69647 .30902 Mdot
.69663 .16187 Mdot
.69681 .30902 Mdot
.69699 .16187 Mdot
.69717 .30902 Mdot
.69737 .30902 Mdot
.69755 .30902 Mdot
.69772 .45617 Mdot
.69789 .30902 Mdot
.69805 .30902 Mdot
.69821 .30902 Mdot
.69839 .30902 Mdot
.69857 .30902 Mdot
.69875 .30902 Mdot
.69895 .30902 Mdot
.69913 .30902 Mdot
.6993 .30902 Mdot
.69947 .30902 Mdot
.69963 .30902 Mdot
.69979 .30902 Mdot
.69996 .16187 Mdot
.70015 .30902 Mdot
.70032 .16187 Mdot
.70052 .16187 Mdot
.7007 .30902 Mdot
.70088 .30902 Mdot
.70104 .16187 Mdot
.70121 .30902 Mdot
.70137 .16187 Mdot
.70154 .30902 Mdot
.70172 .30902 Mdot
.70189 .30902 Mdot
.70209 .30902 Mdot
.70228 .60332 Mdot
.70245 .30902 Mdot
.70262 .30902 Mdot
.70278 .30902 Mdot
.70294 .30902 Mdot
.70311 .16187 Mdot
.70329 .30902 Mdot
.70346 .30902 Mdot
.70366 .30902 Mdot
.70384 .45617 Mdot
.70402 .30902 Mdot
.70419 .30902 Mdot
.70435 .16187 Mdot
.70451 .30902 Mdot
.70468 .30902 Mdot
.70486 .30902 Mdot
.70503 .45617 Mdot
.70522 .45617 Mdot
.70541 .16187 Mdot
.70559 .30902 Mdot
.70575 .30902 Mdot
.70592 .30902 Mdot
.70608 .16187 Mdot
.70625 .30902 Mdot
.70642 .30902 Mdot
.7066 .30902 Mdot
.70679 .16187 Mdot
.70698 .30902 Mdot
.70716 .30902 Mdot
.70733 .30902 Mdot
.7075 .30902 Mdot
.70765 .30902 Mdot
.70782 .30902 Mdot
.708 .30902 Mdot
.70817 .30902 Mdot
.70836 .30902 Mdot
.70856 .30902 Mdot
.70874 .16187 Mdot
.70891 .30902 Mdot
.70908 .30902 Mdot
.70923 .16187 Mdot
.7094 .30902 Mdot
.70958 .30902 Mdot
.70975 .45617 Mdot
.70994 .16187 Mdot
.71014 .30902 Mdot
.71032 .16187 Mdot
.71049 .30902 Mdot
.71066 .30902 Mdot
.71082 .30902 Mdot
.71098 .45617 Mdot
.71116 .45617 Mdot
.71133 .30902 Mdot
.71152 .30902 Mdot
.71172 .30902 Mdot
.7119 .16187 Mdot
.71207 .30902 Mdot
.71224 .16187 Mdot
.7124 .30902 Mdot
.71256 .45617 Mdot
.71273 .30902 Mdot
.71291 .30902 Mdot
.71309 .45617 Mdot
.71329 .45617 Mdot
.71347 .45617 Mdot
.71364 .01472 Mdot
.71381 .30902 Mdot
.71397 .30902 Mdot
.71413 .16187 Mdot
.71431 .16187 Mdot
.71449 .45617 Mdot
.71466 .45617 Mdot
.71486 .30902 Mdot
.71504 .30902 Mdot
.71522 .30902 Mdot
.71538 .30902 Mdot
.71555 .30902 Mdot
.71571 .30902 Mdot
.71588 .30902 Mdot
.71606 .16187 Mdot
.71623 .16187 Mdot
.71643 .30902 Mdot
.71661 .30902 Mdot
.71679 .30902 Mdot
.71696 .30902 Mdot
.71712 .30902 Mdot
.71728 .30902 Mdot
.71745 .30902 Mdot
.71763 .45617 Mdot
.7178 .45617 Mdot
.718 .16187 Mdot
.71818 .30902 Mdot
.71836 .30902 Mdot
.71852 .30902 Mdot
.71869 .45617 Mdot
.71885 .30902 Mdot
.71902 .16187 Mdot
.71919 .30902 Mdot
.71937 .30902 Mdot
.71956 .16187 Mdot
.71975 .30902 Mdot
.71993 .30902 Mdot
.72009 .30902 Mdot
.72026 .30902 Mdot
.72042 .30902 Mdot
.72059 .30902 Mdot
.72076 .30902 Mdot
.72094 .30902 Mdot
.72113 .45617 Mdot
.72132 .30902 Mdot
.7215 .30902 Mdot
.72167 .30902 Mdot
.72184 .30902 Mdot
.722 .30902 Mdot
.72216 .60332 Mdot
.72234 .30902 Mdot
.72252 .30902 Mdot
.7227 .30902 Mdot
.7229 .16187 Mdot
.72308 .30902 Mdot
.72325 .30902 Mdot
.72342 .30902 Mdot
.72358 .30902 Mdot
.72374 .30902 Mdot
.72392 .45617 Mdot
.7241 .30902 Mdot
.72428 .30902 Mdot
.72448 .30902 Mdot
.72466 .60332 Mdot
.72483 .30902 Mdot
.725 .30902 Mdot
.72516 .16187 Mdot
.72532 .45617 Mdot
.7255 .45617 Mdot
.72568 .30902 Mdot
.72586 .30902 Mdot
.72606 .30902 Mdot
.72624 .30902 Mdot
.72641 .45617 Mdot
.72658 .30902 Mdot
.72674 .30902 Mdot
.7269 .30902 Mdot
.72707 .30902 Mdot
.72725 .30902 Mdot
.72743 .30902 Mdot
.72763 .30902 Mdot
.72781 .30902 Mdot
.72798 .30902 Mdot
.72815 .30902 Mdot
.72831 .30902 Mdot
.72847 .45617 Mdot
.72864 .30902 Mdot
.72882 .30902 Mdot
.729 .16187 Mdot
.7292 .16187 Mdot
.72938 .30902 Mdot
.72956 .45617 Mdot
.72972 .30902 Mdot
.72989 .30902 Mdot
.73005 .30902 Mdot
.73022 .30902 Mdot
.73039 .30902 Mdot
.73057 .30902 Mdot
.73076 .30902 Mdot
.73095 .30902 Mdot
.73112 .30902 Mdot
.73129 .30902 Mdot
.73146 .16187 Mdot
.73162 .60332 Mdot
.73178 .30902 Mdot
.73196 .30902 Mdot
.73214 .30902 Mdot
.73233 .16187 Mdot
.73252 .30902 Mdot
.73269 .16187 Mdot
.73286 .30902 Mdot
.73303 .01472 Mdot
.73319 .30902 Mdot
.73335 .30902 Mdot
.73353 .30902 Mdot
.73371 .30902 Mdot
.73389 .16187 Mdot
.73409 .30902 Mdot
.73427 .30902 Mdot
.73444 .30902 Mdot
.73461 .16187 Mdot
.73477 .30902 Mdot
.73493 .30902 Mdot
.73511 .30902 Mdot
.73528 .30902 Mdot
.73547 .16187 Mdot
.73567 .16187 Mdot
.73585 .60332 Mdot
.73602 .30902 Mdot
.73619 .01472 Mdot
.73635 .60332 Mdot
.73651 .16187 Mdot
.73669 .60332 Mdot
.73686 .30902 Mdot
.73705 .30902 Mdot
.73725 .30902 Mdot
.73743 .45617 Mdot
.7376 .45617 Mdot
.73777 .30902 Mdot
.73793 .30902 Mdot
.73809 .16187 Mdot
.73826 .30902 Mdot
.73844 .45617 Mdot
.73862 .30902 Mdot
.73882 .30902 Mdot
.739 .30902 Mdot
.73918 .30902 Mdot
.73934 .45617 Mdot
.7395 .30902 Mdot
.73967 .30902 Mdot
.73984 .30902 Mdot
.74002 .30902 Mdot
.7402 .45617 Mdot
.7404 .30902 Mdot
.74058 .30902 Mdot
.74075 .30902 Mdot
.74092 .30902 Mdot
.74108 .30902 Mdot
.74124 .30902 Mdot
.74141 .30902 Mdot
.74159 .30902 Mdot
.74177 .45617 Mdot
.74197 .30902 Mdot
.74215 .60332 Mdot
.74232 .30902 Mdot
.74249 .30902 Mdot
.74265 .16187 Mdot
.74281 .30902 Mdot
.74298 .30902 Mdot
.74316 .30902 Mdot
.74334 .45617 Mdot
.74353 .45617 Mdot
.74372 .30902 Mdot
.74389 .30902 Mdot
.74406 .30902 Mdot
.74423 .30902 Mdot
.74438 .45617 Mdot
.74455 .45617 Mdot
.74473 .45617 Mdot
.7449 .45617 Mdot
.74509 .30902 Mdot
.74528 .30902 Mdot
.74546 .60332 Mdot
.74563 .30902 Mdot
.7458 .30902 Mdot
.74596 .30902 Mdot
.74612 .30902 Mdot
.7463 .30902 Mdot
.74647 .30902 Mdot
.74666 .30902 Mdot
.74686 .30902 Mdot
.74703 .60332 Mdot
.7472 .30902 Mdot
.74737 .30902 Mdot
.74753 .60332 Mdot
.7477 .30902 Mdot
.74787 .30902 Mdot
.74805 .45617 Mdot
.74823 .16187 Mdot
.74843 .30902 Mdot
.74861 .01472 Mdot
.74878 .30902 Mdot
.74895 .30902 Mdot
.74911 .30902 Mdot
.74928 .30902 Mdot
.74945 .45617 Mdot
.74963 .30902 Mdot
.74981 .30902 Mdot
.75001 .30902 Mdot
.75019 .16187 Mdot
.75036 .30902 Mdot
.75053 .30902 Mdot
.75069 .30902 Mdot
.75085 .45617 Mdot
.75103 .30902 Mdot
.75121 .30902 Mdot
.75139 .30902 Mdot
.75159 .30902 Mdot
.75177 .30902 Mdot
.75194 .30902 Mdot
.75211 .30902 Mdot
.75227 .30902 Mdot
.75243 .30902 Mdot
.75261 .16187 Mdot
.75279 .16187 Mdot
.75296 .30902 Mdot
.75317 .45617 Mdot
.75335 .30902 Mdot
.75352 .30902 Mdot
.75369 .30902 Mdot
.75385 .30902 Mdot
.75401 .30902 Mdot
.75418 .45617 Mdot
.75436 .30902 Mdot
.75454 .30902 Mdot
.75474 .30902 Mdot
.75492 .30902 Mdot
.75509 .16187 Mdot
.75526 .30902 Mdot
.75542 .45617 Mdot
.75558 .30902 Mdot
.75575 .30902 Mdot
.75593 .30902 Mdot
.7561 .30902 Mdot
.7563 .30902 Mdot
.75649 .30902 Mdot
.75666 .30902 Mdot
.75683 .45617 Mdot
.75699 .30902 Mdot
.75715 .60332 Mdot
.75732 .30902 Mdot
.7575 .30902 Mdot
.75767 .30902 Mdot
.75786 .30902 Mdot
.75805 .45617 Mdot
.75823 .30902 Mdot
.7584 .30902 Mdot
.75856 .30902 Mdot
.75872 .30902 Mdot
.75889 .30902 Mdot
.75907 .30902 Mdot
.75924 .16187 Mdot
.75943 .30902 Mdot
.75962 .30902 Mdot
.7598 .30902 Mdot
.75997 .30902 Mdot
.76014 .60332 Mdot
.7603 .30902 Mdot
.76046 .30902 Mdot
.76064 .30902 Mdot
.76081 .30902 Mdot
.761 .30902 Mdot
.7612 .30902 Mdot
.76138 .60332 Mdot
.76155 .16187 Mdot
.76172 .45617 Mdot
.76188 .30902 Mdot
.76204 .45617 Mdot
.76222 .16187 Mdot
.7624 .30902 Mdot
.76258 .30902 Mdot
.76278 .30902 Mdot
.76296 .60332 Mdot
.76313 .30902 Mdot
.7633 .16187 Mdot
.76346 .30902 Mdot
.76362 .16187 Mdot
.7638 .30902 Mdot
.76398 .30902 Mdot
.76416 .45617 Mdot
.76436 .30902 Mdot
.76454 .30902 Mdot
.76471 .30902 Mdot
.76488 .30902 Mdot
.76504 .30902 Mdot
.7652 .30902 Mdot
.76537 .45617 Mdot
.76555 .16187 Mdot
.76573 .45617 Mdot
.76593 .16187 Mdot
.76611 .30902 Mdot
.76629 .30902 Mdot
.76645 .30902 Mdot
.76662 .30902 Mdot
.76678 .45617 Mdot
.76695 .16187 Mdot
.76713 .30902 Mdot
.7673 .30902 Mdot
.7675 .60332 Mdot
.76769 .16187 Mdot
.76786 .30902 Mdot
.76803 .30902 Mdot
.76819 .30902 Mdot
.76835 .45617 Mdot
.76852 .01472 Mdot
.7687 .30902 Mdot
.76887 .30902 Mdot
.76907 .45617 Mdot
.76926 .30902 Mdot
.76943 .30902 Mdot
.7696 .30902 Mdot
.76976 .30902 Mdot
.76992 .30902 Mdot
.77009 .45617 Mdot
.77027 .16187 Mdot
.77044 .30902 Mdot
.77064 .30902 Mdot
.77082 .60332 Mdot
.771 .30902 Mdot
.77117 .30902 Mdot
.77133 .30902 Mdot
.77149 .01472 Mdot
.77166 .30902 Mdot
.77184 .45617 Mdot
.77201 .45617 Mdot
.7722 .30902 Mdot
.77239 .30902 Mdot
.77257 .16187 Mdot
.77274 .30902 Mdot
.77291 .45617 Mdot
.77306 .30902 Mdot
.77323 .16187 Mdot
.77341 .30902 Mdot
.77358 .45617 Mdot
.77377 .30902 Mdot
.77397 .30902 Mdot
.77414 .16187 Mdot
.77431 .45617 Mdot
.77448 .30902 Mdot
.77464 .30902 Mdot
.77481 .30902 Mdot
.77498 .30902 Mdot
.77516 .16187 Mdot
.77534 .45617 Mdot
.77554 .30902 Mdot
.77572 .30902 Mdot
.77589 .30902 Mdot
.77606 .30902 Mdot
.77622 .30902 Mdot
.77639 .01472 Mdot
.77656 .30902 Mdot
.77674 .30902 Mdot
.77692 .30902 Mdot
.77712 .30902 Mdot
.7773 .30902 Mdot
.77747 .30902 Mdot
.77764 .16187 Mdot
.7778 .30902 Mdot
.77797 .30902 Mdot
.77814 .30902 Mdot
.77832 .30902 Mdot
.7785 .16187 Mdot
.7787 .30902 Mdot
.77888 .01472 Mdot
.77905 .30902 Mdot
.77922 .30902 Mdot
.77938 .30902 Mdot
.77954 .01472 Mdot
.77971 .30902 Mdot
.77989 .30902 Mdot
.78007 .30902 Mdot
.78027 .45617 Mdot
.78045 .30902 Mdot
.78063 .30902 Mdot
.78079 .16187 Mdot
.78096 .30902 Mdot
.78112 .16187 Mdot
.78129 .30902 Mdot
.78146 .30902 Mdot
.78164 .16187 Mdot
.78184 .30902 Mdot
.78202 .01472 Mdot
.7822 .60332 Mdot
.78237 .16187 Mdot
.78253 .30902 Mdot
.78269 .16187 Mdot
.78286 .16187 Mdot
.78304 .30902 Mdot
.78321 .30902 Mdot
.78341 .16187 Mdot
.78359 .30902 Mdot
.78377 .16187 Mdot
.78393 .30902 Mdot
.7841 .30902 Mdot
.78426 .30902 Mdot
.78443 .30902 Mdot
.7846 .30902 Mdot
.78478 .16187 Mdot
.78497 .30902 Mdot
.78516 .16187 Mdot
.78533 .30902 Mdot
.7855 .30902 Mdot
.78567 .30902 Mdot
.78583 .30902 Mdot
.786 .30902 Mdot
.78617 .30902 Mdot
.78635 .30902 Mdot
.78653 .45617 Mdot
.78673 .16187 Mdot
.78691 .45617 Mdot
.78708 .30902 Mdot
.78725 .30902 Mdot
.78741 .45617 Mdot
.78757 .30902 Mdot
.78775 .16187 Mdot
.78793 .30902 Mdot
.78811 .30902 Mdot
.78831 .16187 Mdot
.78849 .01472 Mdot
.78866 .30902 Mdot
.78883 .30902 Mdot
.78899 .60332 Mdot
.78915 .30902 Mdot
.78933 .30902 Mdot
.78951 .30902 Mdot
.78969 .30902 Mdot
.78989 .60332 Mdot
.79007 .30902 Mdot
.79024 .60332 Mdot
.79041 .30902 Mdot
.79057 .30902 Mdot
.79073 .30902 Mdot
.79091 .30902 Mdot
.79109 .30902 Mdot
.79127 .30902 Mdot
.79147 .16187 Mdot
.79165 .45617 Mdot
.79182 .30902 Mdot
.79199 .30902 Mdot
.79215 .30902 Mdot
.79231 .30902 Mdot
.79248 .30902 Mdot
.79266 .30902 Mdot
.79284 .45617 Mdot
.79304 .45617 Mdot
.79322 .30902 Mdot
.79339 .30902 Mdot
.79356 .30902 Mdot
.79372 .30902 Mdot
.79388 .30902 Mdot
.79405 .30902 Mdot
.79423 .16187 Mdot
.79441 .30902 Mdot
.79461 .30902 Mdot
.79479 .60332 Mdot
.79497 .30902 Mdot
.79513 .30902 Mdot
.7953 .45617 Mdot
.79546 .30902 Mdot
.79562 .30902 Mdot
.7958 .30902 Mdot
.79598 .30902 Mdot
.79617 .16187 Mdot
.79636 .30902 Mdot
.79653 .30902 Mdot
.7967 .30902 Mdot
.79687 .16187 Mdot
.79703 .30902 Mdot
.79719 .60332 Mdot
.79737 .16187 Mdot
.79755 .16187 Mdot
.79774 .45617 Mdot
.79793 .30902 Mdot
.7981 .30902 Mdot
.79827 .30902 Mdot
.79844 .30902 Mdot
.7986 .45617 Mdot
.79876 .30902 Mdot
.79894 .16187 Mdot
.79912 .30902 Mdot
.7993 .45617 Mdot
.7995 .30902 Mdot
.79968 .30902 Mdot
.79984 .30902 Mdot
.80001 .30902 Mdot
.80017 .16187 Mdot
.80034 .60332 Mdot
.80051 .30902 Mdot
.80069 .30902 Mdot
.80087 .30902 Mdot
.80108 .30902 Mdot
.80125 .30902 Mdot
.80143 .30902 Mdot
.80159 .30902 Mdot
.80175 .30902 Mdot
.80192 .30902 Mdot
.80209 .45617 Mdot
.80227 .16187 Mdot
.80245 .30902 Mdot
.80266 .30902 Mdot
.80283 .30902 Mdot
.80301 .30902 Mdot
.80317 .30902 Mdot
.80333 .30902 Mdot
.8035 .30902 Mdot
.80367 .45617 Mdot
.80385 .30902 Mdot
.80403 .30902 Mdot
.80423 .30902 Mdot
.80441 .30902 Mdot
.80458 .30902 Mdot
.80475 .16187 Mdot
.80491 .30902 Mdot
.80507 .30902 Mdot
.80525 .30902 Mdot
.80543 .45617 Mdot
.80561 .16187 Mdot
.80581 .30902 Mdot
.80599 .30902 Mdot
.80616 .30902 Mdot
.80633 .30902 Mdot
.80649 .30902 Mdot
.80665 .30902 Mdot
.80682 .30902 Mdot
.807 .16187 Mdot
.80718 .30902 Mdot
.80738 .16187 Mdot
.80756 .30902 Mdot
.80773 .30902 Mdot
.8079 .30902 Mdot
.80806 .16187 Mdot
.80822 .45617 Mdot
.80839 .45617 Mdot
.80857 .30902 Mdot
.80875 .45617 Mdot
.80894 .16187 Mdot
.80913 .16187 Mdot
.8093 .30902 Mdot
.80947 .30902 Mdot
.80964 .30902 Mdot
.80979 .30902 Mdot
.80996 .30902 Mdot
.81014 .30902 Mdot
.81031 .30902 Mdot
.81051 .30902 Mdot
.81069 .45617 Mdot
.81087 .30902 Mdot
.81104 .30902 Mdot
.81121 .30902 Mdot
.81136 .45617 Mdot
.81153 .30902 Mdot
.81171 .30902 Mdot
.81188 .30902 Mdot
.81207 .30902 Mdot
.81226 .16187 Mdot
.81244 .30902 Mdot
.81261 .30902 Mdot
.81278 .16187 Mdot
.81294 .30902 Mdot
.8131 .30902 Mdot
.81328 .30902 Mdot
.81346 .30902 Mdot
.81364 .30902 Mdot
.81384 .30902 Mdot
.81402 .45617 Mdot
.81419 .30902 Mdot
.81436 .30902 Mdot
.81452 .30902 Mdot
.81468 .30902 Mdot
.81486 .30902 Mdot
.81504 .30902 Mdot
.81522 .30902 Mdot
.81542 .30902 Mdot
.8156 .30902 Mdot
.81577 .60332 Mdot
.81594 .30902 Mdot
.8161 .30902 Mdot
.81626 .30902 Mdot
.81644 .30902 Mdot
.81662 .30902 Mdot
.8168 .45617 Mdot
.817 .30902 Mdot
.81718 .16187 Mdot
.81735 .30902 Mdot
.81752 .30902 Mdot
.81768 .30902 Mdot
.81784 .16187 Mdot
.81802 .30902 Mdot
.8182 .45617 Mdot
.81838 .30902 Mdot
.81858 .30902 Mdot
.81876 .30902 Mdot
.81893 .45617 Mdot
.8191 .30902 Mdot
.81926 .30902 Mdot
.81942 .30902 Mdot
.81959 .45617 Mdot
.81977 .16187 Mdot
.81995 .30902 Mdot
.82015 .45617 Mdot
.82033 .30902 Mdot
.8205 .30902 Mdot
.82067 .30902 Mdot
.82083 .16187 Mdot
.82099 .30902 Mdot
.82116 .30902 Mdot
.82134 .45617 Mdot
.82152 .16187 Mdot
.82171 .45617 Mdot
.8219 .16187 Mdot
.82207 .60332 Mdot
.82224 .30902 Mdot
.8224 .30902 Mdot
.82256 .30902 Mdot
.82273 .30902 Mdot
.82291 .30902 Mdot
.82309 .16187 Mdot
.82328 .01472 Mdot
.82346 .16187 Mdot
.82364 .45617 Mdot
.82381 .30902 Mdot
.82398 .30902 Mdot
.82413 .01472 Mdot
.8243 .16187 Mdot
.82448 .16187 Mdot
.82465 .30902 Mdot
.82484 .30902 Mdot
.82503 .30902 Mdot
.82521 .30902 Mdot
.82538 .30902 Mdot
.82555 .30902 Mdot
.82571 .30902 Mdot
.82587 .30902 Mdot
.82605 .30902 Mdot
.82622 .30902 Mdot
.82641 .16187 Mdot
.82661 .30902 Mdot
.82679 .30902 Mdot
.82696 .30902 Mdot
.82713 .45617 Mdot
.82728 .30902 Mdot
.82745 .30902 Mdot
.82763 .30902 Mdot
.8278 .45617 Mdot
.82799 .45617 Mdot
.82819 .45617 Mdot
.82837 .30902 Mdot
.82854 .30902 Mdot
.82871 .30902 Mdot
.82887 .30902 Mdot
.82903 .30902 Mdot
.8292 .16187 Mdot
.82938 .30902 Mdot
.82957 .30902 Mdot
.82977 .30902 Mdot
.82995 .30902 Mdot
.83012 .30902 Mdot
.83029 .30902 Mdot
.83045 .30902 Mdot
.83061 .30902 Mdot
.83078 .45617 Mdot
.83096 .30902 Mdot
.83114 .16187 Mdot
.83134 .16187 Mdot
.83152 .45617 Mdot
.83169 .30902 Mdot
.83186 .30902 Mdot
.83202 .16187 Mdot
.83218 .30902 Mdot
.83236 .30902 Mdot
.83254 .30902 Mdot
.83271 .30902 Mdot
.83291 .30902 Mdot
.83309 .30902 Mdot
.83327 .30902 Mdot
.83344 .30902 Mdot
.8336 .30902 Mdot
.83376 .16187 Mdot
.83393 .45617 Mdot
.83411 .30902 Mdot
.83428 .30902 Mdot
.83448 .30902 Mdot
.83467 .30902 Mdot
.83484 .30902 Mdot
.83501 .30902 Mdot
.83517 .30902 Mdot
.83533 .30902 Mdot
.8355 .45617 Mdot
.83568 .45617 Mdot
.83585 .30902 Mdot
.83605 .30902 Mdot
.83623 .30902 Mdot
.83641 .30902 Mdot
.83658 .30902 Mdot
.83674 .30902 Mdot
.8369 .45617 Mdot
.83707 .45617 Mdot
.83725 .16187 Mdot
.83742 .30902 Mdot
.83761 .16187 Mdot
.8378 .45617 Mdot
.83798 .45617 Mdot
.83815 .30902 Mdot
.83831 .30902 Mdot
.83847 .01472 Mdot
.83864 .45617 Mdot
.83881 .45617 Mdot
.83899 .30902 Mdot
.83918 .30902 Mdot
.83937 .16187 Mdot
.83955 .30902 Mdot
.83972 .30902 Mdot
.83989 .30902 Mdot
.84005 .30902 Mdot
.84021 .60332 Mdot
.84039 .45617 Mdot
.84057 .16187 Mdot
.84075 .45617 Mdot
.84095 .45617 Mdot
.84113 .30902 Mdot
.8413 .45617 Mdot
.84147 .30902 Mdot
.84163 .30902 Mdot
.84179 .30902 Mdot
.84197 .30902 Mdot
.84215 .30902 Mdot
.84233 .45617 Mdot
.84253 .30902 Mdot
.84271 .30902 Mdot
.84288 .30902 Mdot
.84305 .30902 Mdot
.84321 .45617 Mdot
.84337 .16187 Mdot
.84355 .30902 Mdot
.84373 .16187 Mdot
.84391 .30902 Mdot
.84411 .30902 Mdot
.84429 .16187 Mdot
.84446 .45617 Mdot
.84463 .30902 Mdot
.84479 .30902 Mdot
.84495 .30902 Mdot
.84512 .45617 Mdot
.8453 .16187 Mdot
.84548 .16187 Mdot
.84568 .30902 Mdot
.84586 .45617 Mdot
.84604 .45617 Mdot
.8462 .30902 Mdot
.84636 .45617 Mdot
.84653 .30902 Mdot
.8467 .30902 Mdot
.84687 .30902 Mdot
.84705 .30902 Mdot
.84725 .30902 Mdot
.84743 .30902 Mdot
.84761 .45617 Mdot
.84777 .30902 Mdot
.84794 .30902 Mdot
.8481 .30902 Mdot
.84827 .30902 Mdot
.84844 .30902 Mdot
.84862 .16187 Mdot
.84882 .16187 Mdot
.849 .16187 Mdot
.84918 .30902 Mdot
.84934 .30902 Mdot
.84951 .30902 Mdot
.84967 .30902 Mdot
.84984 .45617 Mdot
.85001 .30902 Mdot
.85019 .30902 Mdot
.85038 .16187 Mdot
.85057 .30902 Mdot
.85074 .30902 Mdot
.85091 .16187 Mdot
.85108 .30902 Mdot
.85124 .16187 Mdot
.85141 .30902 Mdot
.85158 .30902 Mdot
.85176 .30902 Mdot
.85194 .30902 Mdot
.85214 .30902 Mdot
.85232 .60332 Mdot
.85249 .30902 Mdot
.85266 .30902 Mdot
.85281 .60332 Mdot
.85298 .30902 Mdot
.85316 .16187 Mdot
.85333 .30902 Mdot
.85352 .30902 Mdot
.85372 .45617 Mdot
.8539 .45617 Mdot
.85407 .30902 Mdot
.85424 .30902 Mdot
.8544 .30902 Mdot
.85456 .16187 Mdot
.85474 .30902 Mdot
.85491 .30902 Mdot
.8551 .30902 Mdot
.8553 .30902 Mdot
.85548 .30902 Mdot
.85565 .30902 Mdot
.85582 .16187 Mdot
.85598 .60332 Mdot
.85614 .60332 Mdot
.85631 .30902 Mdot
.85649 .45617 Mdot
.85667 .30902 Mdot
.85688 .16187 Mdot
.85705 .60332 Mdot
.85723 .30902 Mdot
.85739 .45617 Mdot
.85756 .01472 Mdot
.85772 .30902 Mdot
.85789 .60332 Mdot
.85807 .30902 Mdot
.85825 .45617 Mdot
.85845 .30902 Mdot
.85863 .30902 Mdot
.8588 .30902 Mdot
.85897 .30902 Mdot
.85913 .16187 Mdot
.85929 .30902 Mdot
.85946 .30902 Mdot
.85964 .30902 Mdot
.85982 .30902 Mdot
.86002 .30902 Mdot
.8602 .30902 Mdot
.86037 .16187 Mdot
.86054 .30902 Mdot
.86071 .30902 Mdot
.86086 .16187 Mdot
.86103 .30902 Mdot
.86121 .16187 Mdot
.86139 .30902 Mdot
.86158 .16187 Mdot
.86177 .30902 Mdot
.86194 .45617 Mdot
.86211 .30902 Mdot
.86228 .30902 Mdot
.86244 .01472 Mdot
.8626 .30902 Mdot
.86278 .30902 Mdot
.86296 .45617 Mdot
.86315 .45617 Mdot
.86334 .30902 Mdot
.86351 .30902 Mdot
.86368 .30902 Mdot
.86385 .30902 Mdot
.86401 .30902 Mdot
.86417 .30902 Mdot
.86435 .30902 Mdot
.86453 .16187 Mdot
.86471 .60332 Mdot
.86491 .30902 Mdot
.86508 .16187 Mdot
.86525 .30902 Mdot
.86542 .30902 Mdot
.86558 .30902 Mdot
.86575 .30902 Mdot
.86592 .30902 Mdot
.8661 .30902 Mdot
.86628 .30902 Mdot
.86648 .16187 Mdot
.86666 .30902 Mdot
.86683 .30902 Mdot
.867 .30902 Mdot
.86716 .01472 Mdot
.86733 .30902 Mdot
.8675 .45617 Mdot
.86768 .16187 Mdot
.86786 .30902 Mdot
.86806 .30902 Mdot
.86824 .60332 Mdot
.86841 .30902 Mdot
.86858 .30902 Mdot
.86874 .30902 Mdot
.86891 .16187 Mdot
.86908 .30902 Mdot
.86926 .30902 Mdot
.86944 .30902 Mdot
.86964 .45617 Mdot
.86982 .01472 Mdot
.86999 .30902 Mdot
.87016 .30902 Mdot
.87032 .30902 Mdot
.87048 .30902 Mdot
.87066 .16187 Mdot
.87084 .30902 Mdot
.87102 .30902 Mdot
.87122 .45617 Mdot
.8714 .30902 Mdot
.87157 .16187 Mdot
.87174 .30902 Mdot
.8719 .30902 Mdot
.87206 .30902 Mdot
.87223 .30902 Mdot
.87241 .45617 Mdot
.87259 .30902 Mdot
.87279 .45617 Mdot
.87297 .30902 Mdot
.87314 .30902 Mdot
.87331 .16187 Mdot
.87347 .60332 Mdot
.87363 .30902 Mdot
.8738 .16187 Mdot
.87398 .45617 Mdot
.87416 .16187 Mdot
.87436 .30902 Mdot
.87454 .45617 Mdot
.87471 .30902 Mdot
.87488 .30902 Mdot
.87505 .16187 Mdot
.87521 .30902 Mdot
.87537 .30902 Mdot
.87555 .30902 Mdot
.87573 .30902 Mdot
.87592 .16187 Mdot
.87611 .30902 Mdot
.87628 .45617 Mdot
.87645 .16187 Mdot
.87662 .30902 Mdot
.87678 .60332 Mdot
.87694 .30902 Mdot
.87712 .30902 Mdot
.87729 .30902 Mdot
.87748 .16187 Mdot
.87768 .45617 Mdot
.87785 .30902 Mdot
.87802 .30902 Mdot
.87819 .30902 Mdot
.87835 .30902 Mdot
.87851 .30902 Mdot
.87869 .30902 Mdot
.87887 .45617 Mdot
.87905 .30902 Mdot
.87925 .16187 Mdot
.87943 .30902 Mdot
.8796 .30902 Mdot
.87977 .30902 Mdot
.87993 .45617 Mdot
.88009 .16187 Mdot
.88027 .30902 Mdot
.88045 .30902 Mdot
.88063 .30902 Mdot
.88083 .30902 Mdot
.88101 .45617 Mdot
.88118 .30902 Mdot
.88135 .30902 Mdot
.88151 .30902 Mdot
.88167 .30902 Mdot
.88185 .30902 Mdot
.88203 .16187 Mdot
.88221 .30902 Mdot
.88241 .30902 Mdot
.88259 .45617 Mdot
.88276 .01472 Mdot
.88293 .45617 Mdot
.88309 .30902 Mdot
.88325 .30902 Mdot
.88342 .30902 Mdot
.8836 .45617 Mdot
.88378 .16187 Mdot
.88398 .16187 Mdot
.88416 .30902 Mdot
.88434 .30902 Mdot
.8845 .30902 Mdot
.88467 .01472 Mdot
.88483 .45617 Mdot
.885 .30902 Mdot
.88518 .30902 Mdot
.88536 .30902 Mdot
.88556 .16187 Mdot
.88574 .30902 Mdot
.88591 .30902 Mdot
.88608 .30902 Mdot
.88624 .45617 Mdot
.8864 .45617 Mdot
.88657 .16187 Mdot
.88675 .45617 Mdot
.88693 .45617 Mdot
.88713 .30902 Mdot
.88731 .30902 Mdot
.88748 .16187 Mdot
.88765 .30902 Mdot
.88781 .30902 Mdot
.88797 .16187 Mdot
.88814 .30902 Mdot
.88832 .45617 Mdot
.8885 .30902 Mdot
.88869 .16187 Mdot
.88888 .30902 Mdot
.88905 .30902 Mdot
.88922 .30902 Mdot
.88939 .30902 Mdot
.88954 .30902 Mdot
.88971 .30902 Mdot
.88989 .30902 Mdot
.89006 .60332 Mdot
.89025 .30902 Mdot
.89044 .30902 Mdot
.89062 .30902 Mdot
.89079 .16187 Mdot
.89096 .16187 Mdot
.89111 .16187 Mdot
.89128 .45617 Mdot
.89146 .30902 Mdot
.89163 .45617 Mdot
.89182 .16187 Mdot
.89202 .45617 Mdot
.89219 .30902 Mdot
.89236 .30902 Mdot
.89253 .30902 Mdot
.89269 .30902 Mdot
.89286 .30902 Mdot
.89303 .16187 Mdot
.89321 .30902 Mdot
.89339 .30902 Mdot
.89359 .30902 Mdot
.89377 .01472 Mdot
.89394 .30902 Mdot
.89411 .30902 Mdot
.89427 .30902 Mdot
.89444 .45617 Mdot
.89461 .30902 Mdot
.89479 .45617 Mdot
.89497 .45617 Mdot
.89517 .30902 Mdot
.89535 .30902 Mdot
.89552 .30902 Mdot
.89569 .16187 Mdot
.89585 .30902 Mdot
.89602 .45617 Mdot
.89619 .30902 Mdot
.89637 .16187 Mdot
.89655 .16187 Mdot
.89675 .30902 Mdot
.89693 .30902 Mdot
.8971 .30902 Mdot
.89727 .30902 Mdot
.89743 .45617 Mdot
.89759 .01472 Mdot
.89777 .45617 Mdot
.89795 .30902 Mdot
.89812 .30902 Mdot
.89832 .45617 Mdot
.8985 .30902 Mdot
.89868 .01472 Mdot
.89884 .30902 Mdot
.89901 .30902 Mdot
.89917 .45617 Mdot
.89934 .30902 Mdot
.89952 .16187 Mdot
.89969 .30902 Mdot
.89989 .60332 Mdot
.90007 .16187 Mdot
.90025 .30902 Mdot
.90042 .30902 Mdot
.90058 .30902 Mdot
.90074 .30902 Mdot
.90091 .45617 Mdot
.90109 .16187 Mdot
.90126 .30902 Mdot
.90146 .30902 Mdot
.90164 .30902 Mdot
.90182 .30902 Mdot
.90198 .30902 Mdot
.90215 .30902 Mdot
.90231 .45617 Mdot
.90248 .30902 Mdot
.90265 .30902 Mdot
.90283 .30902 Mdot
.90302 .30902 Mdot
.90321 .30902 Mdot
.90339 .01472 Mdot
.90355 .30902 Mdot
.90372 .30902 Mdot
.90388 .30902 Mdot
.90405 .01472 Mdot
.90422 .45617 Mdot
.9044 .30902 Mdot
.90459 .30902 Mdot
.90478 .30902 Mdot
.90496 .30902 Mdot
.90513 .30902 Mdot
.9053 .30902 Mdot
.90546 .30902 Mdot
.90562 .30902 Mdot
.9058 .45617 Mdot
.90598 .30902 Mdot
.90616 .45617 Mdot
.90636 .16187 Mdot
.90654 .30902 Mdot
.90671 .30902 Mdot
.90688 .30902 Mdot
.90704 .30902 Mdot
.9072 .16187 Mdot
.90738 .30902 Mdot
.90756 .45617 Mdot
.90774 .30902 Mdot
.90794 .30902 Mdot
.90812 .45617 Mdot
.90829 .30902 Mdot
.90846 .30902 Mdot
.90862 .30902 Mdot
.90878 .30902 Mdot
.90896 .30902 Mdot
.90914 .16187 Mdot
.90932 .45617 Mdot
.90952 .30902 Mdot
.9097 .30902 Mdot
.90987 .30902 Mdot
.91004 .30902 Mdot
.9102 .30902 Mdot
.91036 .45617 Mdot
.91053 .30902 Mdot
.91071 .16187 Mdot
.91089 .30902 Mdot
.91109 .30902 Mdot
.91127 .30902 Mdot
.91144 .16187 Mdot
.91161 .30902 Mdot
.91177 .30902 Mdot
.91193 .30902 Mdot
.9121 .30902 Mdot
.91228 .30902 Mdot
.91246 .45617 Mdot
.91266 .30902 Mdot
.91284 .16187 Mdot
.91302 .45617 Mdot
.91318 .30902 Mdot
.91335 .30902 Mdot
.91351 .16187 Mdot
.91368 .30902 Mdot
.91385 .30902 Mdot
.91403 .30902 Mdot
.91423 .30902 Mdot
.91441 .30902 Mdot
.91459 .30902 Mdot
.91475 .30902 Mdot
.91492 .30902 Mdot
.91508 .30902 Mdot
.91524 .30902 Mdot
.91542 .16187 Mdot
.9156 .45617 Mdot
.91579 .30902 Mdot
.91598 .30902 Mdot
.91615 .30902 Mdot
.91632 .30902 Mdot
.91649 .30902 Mdot
.91665 .30902 Mdot
.91681 .30902 Mdot
.91699 .30902 Mdot
.91717 .30902 Mdot
.91735 .30902 Mdot
.91755 .30902 Mdot
.91773 .30902 Mdot
.91789 .16187 Mdot
.91806 .60332 Mdot
.91822 .30902 Mdot
.91839 .30902 Mdot
.91856 .30902 Mdot
.91874 .45617 Mdot
.91892 .30902 Mdot
.91913 .45617 Mdot
.9193 .30902 Mdot
.91948 .30902 Mdot
.91964 .30902 Mdot
.9198 .30902 Mdot
.91997 .30902 Mdot
.92014 .30902 Mdot
.92032 .45617 Mdot
.9205 .30902 Mdot
.92071 .30902 Mdot
.92088 .16187 Mdot
.92106 .30902 Mdot
.92122 .30902 Mdot
.92138 .30902 Mdot
.92155 .30902 Mdot
.92172 .30902 Mdot
.9219 .45617 Mdot
.92208 .30902 Mdot
.92228 .60332 Mdot
.92246 .30902 Mdot
.92264 .16187 Mdot
.9228 .01472 Mdot
.92296 .45617 Mdot
.92313 .30902 Mdot
.9233 .45617 Mdot
.92348 .30902 Mdot
.92366 .16187 Mdot
.92386 .30902 Mdot
.92404 .30902 Mdot
.92421 .30902 Mdot
.92438 .30902 Mdot
.92454 .16187 Mdot
.9247 .45617 Mdot
.92487 .01472 Mdot
.92505 .30902 Mdot
.92523 .45617 Mdot
.92543 .30902 Mdot
.92561 .45617 Mdot
.92578 .30902 Mdot
.92595 .30902 Mdot
.92612 .16187 Mdot
.92627 .45617 Mdot
.92644 .30902 Mdot
.92662 .30902 Mdot
.9268 .30902 Mdot
.927 .16187 Mdot
.92718 .30902 Mdot
.92735 .30902 Mdot
.92752 .30902 Mdot
.92769 .30902 Mdot
.92785 .30902 Mdot
.92801 .30902 Mdot
.92819 .30902 Mdot
.92837 .16187 Mdot
.92856 .16187 Mdot
.92875 .30902 Mdot
.92892 .30902 Mdot
.92909 .30902 Mdot
.92926 .30902 Mdot
.92942 .30902 Mdot
.92958 .16187 Mdot
.92976 .45617 Mdot
.92994 .16187 Mdot
.93012 .30902 Mdot
.93032 .01472 Mdot
.93049 .45617 Mdot
.93066 .30902 Mdot
.93083 .30902 Mdot
.93099 .30902 Mdot
.93115 .16187 Mdot
.93133 .30902 Mdot
.93151 .30902 Mdot
.93169 .30902 Mdot
.93189 .30902 Mdot
.93207 .30902 Mdot
.93224 .30902 Mdot
.93241 .30902 Mdot
.93257 .30902 Mdot
.93273 .30902 Mdot
.93291 .30902 Mdot
.93309 .01472 Mdot
.93327 .30902 Mdot
.93347 .30902 Mdot
.93365 .30902 Mdot
.93382 .30902 Mdot
.93399 .30902 Mdot
.93415 .30902 Mdot
.93431 .30902 Mdot
.93449 .16187 Mdot
.93467 .45617 Mdot
.93485 .16187 Mdot
.93505 .16187 Mdot
.93523 .30902 Mdot
.9354 .30902 Mdot
.93557 .30902 Mdot
.93573 .30902 Mdot
.93589 .30902 Mdot
.93607 .30902 Mdot
.93625 .30902 Mdot
.93643 .30902 Mdot
.93663 .30902 Mdot
.93681 .01472 Mdot
.93698 .30902 Mdot
.93715 .30902 Mdot
.93731 .30902 Mdot
.93747 .45617 Mdot
.93764 .16187 Mdot
.93782 .30902 Mdot
.938 .30902 Mdot
.9382 .30902 Mdot
.93838 .30902 Mdot
.93855 .30902 Mdot
.93872 .30902 Mdot
.93888 .30902 Mdot
.93904 .30902 Mdot
.93921 .30902 Mdot
.93939 .45617 Mdot
.93957 .45617 Mdot
.93977 .01472 Mdot
.93995 .60332 Mdot
.94012 .16187 Mdot
.94029 .30902 Mdot
.94046 .16187 Mdot
.94062 .45617 Mdot
.94078 .30902 Mdot
.94096 .30902 Mdot
.94114 .45617 Mdot
.94133 .16187 Mdot
.94152 .16187 Mdot
.94169 .30902 Mdot
.94186 .30902 Mdot
.94203 .30902 Mdot
.94219 .30902 Mdot
.94235 .30902 Mdot
.94253 .30902 Mdot
.9427 .30902 Mdot
.94289 .16187 Mdot
.94308 .60332 Mdot
.94326 .30902 Mdot
.94343 .01472 Mdot
.9436 .30902 Mdot
.94376 .30902 Mdot
.94392 .30902 Mdot
.9441 .30902 Mdot
.94428 .16187 Mdot
.94446 .30902 Mdot
.94466 .30902 Mdot
.94484 .30902 Mdot
.94501 .30902 Mdot
.94518 .30902 Mdot
.94533 .30902 Mdot
.9455 .30902 Mdot
.94567 .30902 Mdot
.94585 .30902 Mdot
.94604 .45617 Mdot
.94624 .30902 Mdot
.94642 .45617 Mdot
.94659 .01472 Mdot
.94676 .45617 Mdot
.94692 .30902 Mdot
.94708 .30902 Mdot
.94725 .30902 Mdot
.94743 .30902 Mdot
.94762 .30902 Mdot
.94782 .30902 Mdot
.948 .30902 Mdot
.94817 .30902 Mdot
.94834 .30902 Mdot
.9485 .30902 Mdot
.94866 .30902 Mdot
.94883 .16187 Mdot
.94901 .30902 Mdot
.94919 .30902 Mdot
.94939 .30902 Mdot
.94957 .30902 Mdot
.94974 .30902 Mdot
.94991 .30902 Mdot
.95007 .30902 Mdot
.95023 .30902 Mdot
.95041 .16187 Mdot
.95059 .45617 Mdot
.95077 .30902 Mdot
.95097 .30902 Mdot
.95115 .30902 Mdot
.95132 .30902 Mdot
.95149 .30902 Mdot
.95165 .30902 Mdot
.95181 .45617 Mdot
.95198 .45617 Mdot
.95216 .45617 Mdot
.95233 .16187 Mdot
.95253 .45617 Mdot
.95272 .16187 Mdot
.95289 .30902 Mdot
.95306 .30902 Mdot
.95322 .30902 Mdot
.95338 .45617 Mdot
.95355 .30902 Mdot
.95373 .30902 Mdot
.9539 .16187 Mdot
.9541 .30902 Mdot
.95428 .30902 Mdot
.95446 .60332 Mdot
.95463 .30902 Mdot
.95479 .30902 Mdot
.95495 .30902 Mdot
.95512 .16187 Mdot
.9553 .30902 Mdot
.95547 .45617 Mdot
.95566 .30902 Mdot
.95585 .30902 Mdot
.95603 .30902 Mdot
.9562 .30902 Mdot
.95637 .30902 Mdot
.95652 .30902 Mdot
.95669 .30902 Mdot
.95687 .30902 Mdot
.95704 .16187 Mdot
.95723 .16187 Mdot
.95742 .45617 Mdot
.9576 .30902 Mdot
.95777 .30902 Mdot
.95794 .30902 Mdot
.9581 .30902 Mdot
.95826 .30902 Mdot
.95844 .45617 Mdot
.95862 .30902 Mdot
.9588 .30902 Mdot
.959 .16187 Mdot
.95918 .45617 Mdot
.95935 .16187 Mdot
.95952 .30902 Mdot
.95968 .30902 Mdot
.95984 .30902 Mdot
.96002 .30902 Mdot
.9602 .30902 Mdot
.96038 .16187 Mdot
.96058 .30902 Mdot
.96076 .30902 Mdot
.96093 .30902 Mdot
.9611 .30902 Mdot
.96126 .30902 Mdot
.96142 .30902 Mdot
.9616 .45617 Mdot
.96178 .30902 Mdot
.96196 .30902 Mdot
.96216 .30902 Mdot
.96234 .30902 Mdot
.96251 .30902 Mdot
.96268 .30902 Mdot
.96284 .30902 Mdot
.963 .30902 Mdot
.96317 .30902 Mdot
.96335 .30902 Mdot
.96353 .45617 Mdot
.96373 .30902 Mdot
.96391 .30902 Mdot
.96409 .30902 Mdot
.96425 .16187 Mdot
.96442 .30902 Mdot
.96458 .01472 Mdot
.96475 .30902 Mdot
.96493 .45617 Mdot
.9651 .45617 Mdot
.9653 .01472 Mdot
.96548 .60332 Mdot
.96566 .30902 Mdot
.96582 .30902 Mdot
.96599 .45617 Mdot
.96615 .30902 Mdot
.96632 .30902 Mdot
.9665 .30902 Mdot
.96667 .16187 Mdot
.96687 .30902 Mdot
.96705 .30902 Mdot
.96723 .30902 Mdot
.96739 .30902 Mdot
.96756 .30902 Mdot
.96772 .30902 Mdot
.96789 .16187 Mdot
.96806 .45617 Mdot
.96824 .30902 Mdot
.96843 .30902 Mdot
.96862 .16187 Mdot
.96879 .30902 Mdot
.96896 .30902 Mdot
.96913 .30902 Mdot
.96929 .30902 Mdot
.96946 .45617 Mdot
.96963 .30902 Mdot
.96981 .30902 Mdot
.97 .30902 Mdot
.97019 .30902 Mdot
.97037 .60332 Mdot
.97054 .30902 Mdot
.97071 .01472 Mdot
.97086 .30902 Mdot
.97103 .30902 Mdot
.97121 .30902 Mdot
.97138 .45617 Mdot
.97157 .45617 Mdot
.97177 .30902 Mdot
.97195 .30902 Mdot
.97212 .30902 Mdot
.97229 .30902 Mdot
.97245 .30902 Mdot
.97261 .30902 Mdot
.97279 .30902 Mdot
.97296 .30902 Mdot
.97315 .30902 Mdot
.97335 .30902 Mdot
.97353 .30902 Mdot
.9737 .16187 Mdot
.97387 .30902 Mdot
.97403 .30902 Mdot
.97419 .16187 Mdot
.97436 .30902 Mdot
.97454 .30902 Mdot
.97472 .45617 Mdot
.97493 .30902 Mdot
.97511 .30902 Mdot
.97528 .30902 Mdot
.97544 .30902 Mdot
.97561 .16187 Mdot
.97577 .30902 Mdot
.97594 .45617 Mdot
.97607 .16187 Mdot
.97619 .16187 Mdot
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[32]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00;P3oool00`000000oooo0?oo
o`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool200000?00oooo000^0?ooo`80
000000@0oooo0000003oool0oooo0P0000000`3oool00000000000030?ooo`050000003oool0oooo
0?ooo`0000000P3oool00`000000oooo0?ooo`020?ooo`040000003oool0000000000080oooo0P00
0000103oool000000?ooo`0000020?ooo`<000000`3oool200000080oooo0`0000000`3oool00000
0?ooo`020?ooo`030000003oool0oooo00@000000P3oool00`000000oooo00000003000000040?oo
o`000000oooo0?ooo`800000103oool200000080oooo00L000000?ooo`3oool0oooo0000003oool0
000000<0oooo0P0000000`3oool000000?ooo`020?ooo`800000103oool02@000000oooo00000000
0000oooo0000003oool000000?ooo`0200000080oooo00@000000?ooo`3oool000000P3oool20000
00@0oooo0P000000103oool000000?ooo`3oool2000000030?ooo`000000oooo0080oooo0P000002
0?ooo`030000003oool00000008000000`3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo0080oooo00@000000?ooo`3oool00000103oool4000000030?ooo`000000oooo00H0oooo0P00
00020?ooo`030000003oool0oooo0080oooo100000020?ooo`060000003oool000000000003oool0
00000`3oool7000000<0oooo00<000000?ooo`0000003P3oool002h0oooo00<000000?ooo`3oool0
k`3oool002h0oooo00<000000?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3oool0k`3oool0
00T0oooo0`0000000`3oool000000?ooo`020?ooo`030000003oool0oooo0080oooo1@0000020?oo
o`8000003`3oool00`000000oooo0?ooo`3_0?ooo`002P3oool01`000000oooo0?ooo`3oool00000
0?ooo`0000001`3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0303o
ool00`000000oooo0?ooo`3_0?ooo`001`3oool5000000<0oooo00<000000?ooo`3oool01P3oool0
1@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0303oool00`000000oooo0?oo
o`3_0?ooo`0000<0oooo0000000000000P0000020?ooo`040000003oool0oooo000000<0oooo00<0
00000?ooo`0000001`3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0
303oool300000>l0oooo00080?ooo`030000003oool000000080oooo00D000000?ooo`3oool0oooo
000000060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo0>l0oooo00090?ooo`800000303oool2000000@0oooo0P0000080?ooo`<00000
0P3oool300000?40oooo000X0?ooo`070000003oool0oooo0?ooo`000000oooo0000003a0?ooo`00
8P3oool3000000<0oooo00@000000?ooo`3oool0oooo0`00003a0?ooo`00:03oool01`000000oooo
0?ooo`3oool000000?ooo`000000l@3oool002L0oooo0P0000040?ooo`800000l@3oool002h0oooo
00<000000?ooo`3oool0k`3oool002h0oooo0P00003`0?ooo`00;P3oool00`000000oooo0?ooo`3_
0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00
;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool0
0`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool200000?00
oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^
0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`03
0000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003o
ool0oooo0>l0oooo000^0?ooo`800000l03oool002h0oooo00<000000?ooo`3oool0k`3oool002h0
oooo00<000000?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3oool0k`3oool002h0oooo6@00
00000`3oool0000000000002000000030?ooo`000000000000@000000P3ooolC000000030?ooo`00
0000000004l0000000<0oooo000000000000EP00000=0?ooo`001`3oool400000080oooo00D00000
0?ooo`3oool0oooo000000040?ooo`D000000P3oool2000000l0oooo00<000000?ooo`3oool0k`3o
ool000L0oooo00<000000?ooo`3oool0103oool00`000000oooo000000070?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0>l0oooo0008
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01P3oool01@000000oooo0?ooo`3o
ool000000080oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`3_0?ooo`0000<0oooo
0000000000000P0000040?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000001`3oool0
1@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0303oool300000>l0oooo0007
0?ooo`040000003oool0oooo00000080oooo00D000000?ooo`3oool0oooo000000060?ooo`050000
003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0>l0
oooo00080?ooo`8000003@3oool2000000@0oooo0P0000080?ooo`<000000P3oool300000?40oooo
000X0?ooo`070000003oool0oooo0?ooo`000000oooo0000003a0?ooo`008P3oool3000000<0oooo
00@000000?ooo`3oool0oooo0`00003a0?ooo`00:03oool01`000000oooo0?ooo`3oool000000?oo
o`000000l@3oool002L0oooo0P0000040?ooo`800000l@3oool002h0oooo00<000000?ooo`3oool0
k`3oool002h0oooo0P00003`0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`00
0000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo
0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_
0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool200000?00oooo000^0?ooo`030000
003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0
oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo02l0
oooo0P0000040?ooo`800000103oool2000002L0oooo1@0000020?ooo`800000103oool2000000@0
oooo0P00000T0?ooo`D000000P3oool2000000@0oooo0P0000040?ooo`8000009@3oool4000000<0
oooo0P0000040?ooo`800000103oool2000000<0oooo000^0?ooo`030000003oool0oooo02h0oooo
00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
000002P0oooo00D000000?ooo`3oool0oooo000000020?ooo`040000003oool0oooo00000080oooo
00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`0S0?ooo`050000003oool0oooo0?oo
o`0000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00<0
00000?ooo`3oool08P3oool00`000000oooo0?ooo`030?ooo`040000003oool0oooo00000080oooo
00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`00;P3oool00`000000
oooo0?ooo`0a0?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool0
0`000000oooo0?ooo`0V0?ooo`050000003oool0oooo0?ooo`0000000P3oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool08`3oool00`000000
oooo0?ooo`040?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool0
0`000000oooo0?ooo`0S0?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo000^0?ooo`800000
<03oool2000000<0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00000X0?oo
o`050000003oool0oooo0?ooo`0000000P3oool010000000oooo0?ooo`0000020?ooo`040000003o
ool0oooo00000080oooo00<000000?ooo`3oool08`3oool00`000000oooo0?ooo`020?ooo`800000
0`3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000002H0oooo00D000000?oo
o`3oool0oooo000000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000
0P3oool100000040oooo0@3oool002h0oooo00<000000?ooo`3oool0;`3oool00`000000oooo0?oo
o`020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000:03oool01@000000
oooo0?ooo`3oool000000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`030000003oool0oooo02<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?oo
o`020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000903oool010000000
oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`00;P3oool00`000000oooo0?ooo`0_0?ooo`<000000`3o
ool2000000@0oooo0P00000X0?ooo`800000103oool2000000@0oooo0P0000040?ooo`8000009@3o
ool2000000@0oooo0`0000030?ooo`800000103oool2000002H0oooo0P0000040?ooo`800000103o
ool2000000@0oooo0P0000030?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`00
0000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo
0?ooo`3_0?ooo`00;P3ooooU000000d0oooo000X0?oooo4000001`3oool002h0oooo00<000000?oo
o`3oool0203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool0203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool0203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00X0oooo000^0?ooo`03
0000003oool0oooo03H0oooo00<000000?ooo`3oool0=P3oool00`000000oooo0?ooo`0f0?ooo`03
0000003oool0oooo03L0oooo00<000000?ooo`3oool02P3oool002h0oooo00<000000?ooo`3oool0
k`3oool002h0oooo00<000000?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3oool0k`3oool0
02h0oooo00<000000?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3oool0k`3oool002h0oooo
0P00003`0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_
0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00
;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool0
0`000000oooo0?ooo`3_0?ooo`00;P3oool200000?00oooo000^0?ooo`030000003oool0oooo0>l0
oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^
0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`03
0000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`800000l03o
ool002h0oooo00<000000?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3oool0k`3oool002h0
oooo00<000000?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3oool0k`3oool000L0oooo1000
00020?ooo`050000003oool0oooo0?ooo`000000103oool500000080oooo0P00000?0?ooo`030000
003oool0oooo0>l0oooo00070?ooo`030000003oool0oooo00@0oooo00<000000?ooo`0000001`3o
ool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0303oool00`000000oooo
0?ooo`3_0?ooo`00203oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00H0oooo00D0
00000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0
k`3oool000T0oooo00<000000?ooo`3oool00P3oool00`000000oooo000000070?ooo`050000003o
ool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0<0?ooo`<00000k`3oool000L0oooo00@0
00000?ooo`3oool000000P3oool01@000000oooo0?ooo`3oool0000000H0oooo00D000000?ooo`3o
ool0oooo000000020?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0k`3oool000P0
oooo0P00000=0?ooo`800000103oool2000000P0oooo0`0000020?ooo`<00000l@3oool002P0oooo
00H000000?ooo`3oool0oooo0000003ooolE000000030?ooo`00000000000780000000<0oooo0000
000000005@0000000`3oool0000000000005000000030?ooo`000000000000T0000000<0oooo0000
00000000;000000=0?ooo`008P3oool3000000<0oooo00@000000?ooo`3oool0oooo0`00003a0?oo
o`00:03oool01`000000oooo0?ooo`3oool000000?ooo`000000l@3oool002L0oooo0P0000040?oo
o`800000l@3oool002h0oooo00<000000?ooo`3oool0k`3oool002h0oooo0P00003`0?ooo`00;P3o
ool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`00
0000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo
0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo0?ooo`3_
0?ooo`00;P3oool200000?00oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000
003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0
oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`030000003oool0oooo0>l0
oooo000^0?ooo`030000003oool0oooo0>l0oooo000^0?ooo`800000l03oool002h0oooo00<00000
0?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3oool0k`3oool002h0oooo00<000000?ooo`3o
ool0k`3oool000T0oooo0`0000000`3oool000000?ooo`020?ooo`030000003oool0oooo0080oooo
1@0000020?ooo`8000003`3oool00`000000oooo0?ooo`3_0?ooo`002P3oool01`000000oooo0?oo
o`3oool000000?ooo`0000001`3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?oo
o`3oool0303oool00`000000oooo0?ooo`3_0?ooo`001`3oool5000000<0oooo00<000000?ooo`3o
ool01P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0303oool00`00
0000oooo0?ooo`3_0?ooo`001`3oool010000000oooo0?ooo`0000030?ooo`030000003oool00000
00L0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00`0oooo00<00000
0?ooo`3oool0k`3oool000P0oooo00<000000?ooo`0000000P3oool01@000000oooo0?ooo`3oool0
000000H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00`0oooo0`00
003_0?ooo`002@3oool2000000`0oooo0P0000040?ooo`8000003`3oool00`000000oooo0?ooo`3_
0?ooo`009`3oool300000080oooo0`00003a0?ooo`00:03oool01`000000oooo0?ooo`3oool00000
0?ooo`000000l@3oool00280oooo0`0000030?ooo`040000003oool0oooo0?ooo`<00000l@3oool0
02P0oooo00L000000?ooo`3oool0oooo0000003oool000000?40oooo000W0?ooo`800000103oool2
000000030?ooo`000000000000@0000000@0oooo0000003oool00000203oool4000000<0oooo0P00
00030?ooo`<000000P3oool00`000000oooo000000030?ooo`030000003oool0oooo00<0oooo00@0
00000?ooo`00000000000P3oool00`000000oooo000000040?ooo`<0000000<0oooo0000003oool0
1P3oool00`000000oooo00000003000000060?ooo`00000000000000003oool000000P3oool30000
0080oooo00<000000?ooo`0000000`3oool2000000<0oooo0P0000030?ooo`050000003oool0oooo
0?ooo`0000000P3oool00`000000oooo000000020?ooo`050000003oool000000?ooo`0000000P3o
ool010000000oooo0000000000040?ooo`030000003oool0000000@0oooo00H000000?ooo`000000
oooo0000003oool300000080oooo00<000000?ooo`0000000`3oool00`000000oooo000000040?oo
o`L000000P3oool400000080oooo0P0000000`3oool00000000000030?ooo`8000000`3oool00`00
0000oooo0?ooo`0700000080oooo00@000000?ooo`3oool000000`3oool2000000<0oooo0P000002
0?ooo`050000003oool000000?ooo`0000003P3oool002h0oooo00<000000?ooo`3oool0k`3oool0
02h0oooo0P00003`0?ooo`00;P3oool00`000000oooo0?ooo`3_0?ooo`00;P3oool00`000000oooo
0?ooo`3_0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-407.005, -5.56411*^-16, \
8.75279, 6.28932*^-18}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Methods", "Section",
  CellTags->"c:2"],

Cell[CellGroupData[{

Cell["IDA", "Subsection",
  CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"}],

Cell[TextData[{
  "The ",
  StyleBox["IDA", "MR"],
  " package is part of the ",
  ButtonBox["SUNDIALS",
    ButtonData:>{
      URL[ "http://www.llnl.gov/CASC/sundials/"], None},
    ButtonStyle->"Hyperlink"],
  " (SUite of Nonlinear and DIfferential/ALgebraic equation Solvers) \
developed at the Center for Applied Scientific Computing of Lawrence \
Livermore National Laboratory. As described in the IDA user guide [",
  ButtonBox["HT99",
    ButtonData:>{"References.nb", "HT99"},
    ButtonStyle->"Hyperlink"],
  "], \[OpenCurlyDoubleQuote]IDA is a general purpose solver for the initial \
value problem for systems of differential-algebraic equations (DAEs). The \
name IDA stands for Implicit Differential-Algebraic solver. IDA is based on \
DASPK ...\[CloseCurlyDoubleQuote] ",
  ButtonBox["DASPK",
    ButtonData:>{
      URL[ "http://www.netlib.org/ode/daspk.tgz"], None},
    ButtonStyle->"Hyperlink"],
  " [",
  ButtonBox["BHP94",
    ButtonData:>{"References.nb", "BHP94"},
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["BHP98",
    ButtonData:>{"References.nb", "BHP98"},
    ButtonStyle->"Hyperlink"],
  "] is a FORTRAN code for solving large scale differential-algebraic \
systems."
}], "Text"],

Cell[TextData[{
  "In ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", an interface has been provided to the ",
  StyleBox["IDA", "MR"],
  " package so that rather than needing to write a function in C for \
evaluating the residual and compiling the program, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " generates the function automatically from the equations you input to ",
  StyleBox["NDSolve", "MR"],
  ". "
}], "Text"],

Cell[TextData[{
  "IDA solves the system (1) with Backward Differentiation Formula (BDF) \
methods of orders 1 through 5, implemented using a variable-step form. The \
BDF of order ",
  StyleBox["k",
    FontSlant->"Italic"],
  " is at time ",
  Cell[BoxData[
      \(TraditionalForm\`t\_n = \ t\_\(n - 1\)\  + \ h\_n\)]],
  " is given by the formula"
}], "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{\(\[Sum]\+\(i = 1\)\%k a\_\(n, i\)\ x\_\(n - i\)\), "=", " ", 
        RowBox[{\(h\_n\), " ", 
          RowBox[{
            SuperscriptBox[\(x\_n\), "\[Prime]",
              MultilineFunction->None], "."}]}]}], 
      TraditionalForm]], "DisplayMath"],

Cell[TextData[{
  "The coefficients ",
  Cell[BoxData[
      \(TraditionalForm\`a\_\(n, i\)\)], "DisplayMath"],
  " depend on the order ",
  StyleBox["k",
    FontSlant->"Italic"],
  " and past step sizes. Applying the BDF to the DAE (1) gives a system of \
nonlinear equations to solve:"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`F(t\_n, \ 
        x\_n, \ \(1\/h\_n\) \(\[Sum]\+\(i = 1\)\%k a\_\(n, i\)\ x\_\(n - \
i\)\))\  = \ 0. \)], "DisplayMath"],

Cell["\<\
The solution of the system is achieved by Newton-type methods, typically \
using an approximation to the Jacobian\
\>", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{"J", " ", "=", " ", 
          RowBox[{
            FractionBox[\(\[PartialD]F\), 
              RowBox[{
                FormBox["\[PartialD]",
                  "InputForm"], "x"}]], "+", " ", 
            RowBox[{\(c\_n\), 
              FractionBox[\(\[PartialD]F\), 
                RowBox[{
                  FormBox["\[PartialD]",
                    "InputForm"], \(x'\)}]]}]}]}], " ", ",", 
        " ", \(where\ c\_n\  = \ \(\(\[Alpha]\_\(n, 0\)\/h\_n\)\(.\)\)\)}], 
      TraditionalForm]], "NumberedEquation"],

Cell[TextData[{
  " \[OpenCurlyDoubleQuote]Its [IDAs] most notable feature is that, in the \
solution of the underlying nonlinear system at each time step, it offers a \
choice of Newton/direct methods or an Inexact Newton/Krylov (iterative) \
method.\[CloseCurlyDoubleQuote] [",
  ButtonBox["HT99",
    ButtonData:>{"References.nb", "HT99"},
    ButtonStyle->"Hyperlink"],
  "] In ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", you can access these solvers using method options or use the default ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " ",
  StyleBox["LinearSolve", "MR"],
  ", which switches automatically to direct sparse solvers for large \
problems."
}], "Text"],

Cell[TextData[{
  "At each step of the solution, IDA computes an estimate ",
  Cell[BoxData[
      \(TraditionalForm\`E\_n\)]],
  " of the local truncation error and the step size and order are chosen so \
that the weighted norm ",
  Cell[BoxData[
      \(Norm[E\_n/w\_n]\)], "Input",
    FontWeight->"Plain"],
  "is less than 1. The ",
  Cell[BoxData[
      \(TraditionalForm\`j\^th\)]],
  "component, ",
  Cell[BoxData[
      SubscriptBox[
        StyleBox["w",
          FontSlant->"Italic"], 
        RowBox[{
          StyleBox["n",
            FontSlant->"Italic"], ",", 
          StyleBox["j",
            FontSlant->"Italic"]}]]], "Input",
    FontWeight->"Plain"],
  ", of ",
  Cell[BoxData[
      SubscriptBox[
        StyleBox["w",
          FontSlant->"Italic"], 
        StyleBox["n",
          FontSlant->"Italic"]]], "Input",
    FontWeight->"Plain"],
  " is given by"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`w\_\(n, j\) = \(\(1\/\(10\^\(-prec\)\ \
\[LeftBracketingBar]x\_\(n, j\)\[RightBracketingBar] + 
              10\^\(-acc\)\)\)\(.\)\)\)], "DisplayMath"],

Cell[TextData[{
  "The values ",
  StyleBox["prec", "MR"],
  " and ",
  StyleBox["acc", "MR"],
  " are taken from the ",
  StyleBox["NDSolve", "MR"],
  " settings for the ",
  StyleBox["PrecisionGoal->prec", "MR"],
  " and ",
  StyleBox["AccuracyGoal->", "MR"],
  StyleBox["acc", "MR"],
  ". "
}], "Text"],

Cell[TextData[{
  "Because IDA provides a great deal of flexibility, particularly in the way \
nonlinear equations are solved, there are a number of method options which \
allow you to control how this is done. You can use the method options to IDA \
by giving ",
  StyleBox["NDSolve", "MR"],
  " the option ",
  StyleBox["Method->{IDA, ", "MR"],
  StyleBox["ida method options", "MR",
    FontSlant->"Italic"],
  StyleBox["}", "MR"],
  "."
}], "Text"],

Cell[TextData[{
  "The options for the ",
  StyleBox["IDA", "MR"],
  " method are associated with the symbol ",
  StyleBox["IDA", "MR"],
  " in the ",
  StyleBox["NDSolve`", "MR"],
  " context."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Options[NDSolve`IDA]\)], "Input",
  CellLabel->"In[34]:="],

Cell[BoxData[
    \({"DenseOutput" \[Rule] False, "MaxDifferenceOrder" \[Rule] 5, 
      "ImplicitSolver" \[Rule] "Newton"}\)], "Output",
  CellLabel->"Out[34]="]
}, Open  ]],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["IDA method option name", "SO10"]]], Cell[
            TextData[StyleBox["default value", "SO10"]]], " "},
          {Cell[TextData[StyleBox["\"DenseOutput\"", "MR"]]], 
            StyleBox["False",
              
              "MR"], \(Whether\ to\ have\ the\ output\ function\ give\ the\ \
solution\ at\ intermediate\ points\ to\ the\ full\ order\ of\ the\ \(\(method\
\)\(.\)\)\)},
          {Cell[TextData[StyleBox["\"ImplicitSolver\"", "MR"]]], 
            StyleBox[Cell[TextData[StyleBox["\"Newton\"", "MR"]]],
              "MR"], Cell["How to solve the implicit equations. ", "Text"]},
          {Cell[TextData[StyleBox["\"MaxDifferenceOrder\"", "MR"]]], 
            StyleBox["5",
              "MR"], Cell["The maximum order BDF to use. ", "Text"]}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  StyleBox["IDA", "MR"],
  " method options."
}], "Caption"],

Cell[TextData[{
  "When strict accuracy of intermediate values computed with the ",
  StyleBox["InterpolatingFunction", "MR"],
  " object returned from ",
  StyleBox["NDSolve", "MR"],
  " is important, you will want to use the method option setting ",
  StyleBox["\"DenseOutput\"->True", "MR"],
  ".  By default NDSolve stores a minimal amount of data to represent the \
solution well enough for graphical purposes.   Keeping the amount of data \
small saves on both memory and time for more complicated solutions.  \
Depending on the use you may want to make of the solution, it may be \
appropriate to choose dense output."
}], "Text"],

Cell[TextData[{
  "As an example which highlights the difference between dense and minimal \
output, consider tracking a quantity, ",
  Cell[BoxData[
      \(TraditionalForm\`f(t) = \ \(x(t)\)\^2 + \(y(t)\)\^2\)]],
  " derived from the solution of a simple linear equation where the exact \
solution can be computed using ",
  StyleBox["DSolve", "MR"],
  "."
}], "Text"],

Cell["\<\
This defines the function f giving the quantity as a function of time based \
on solutions x[t] and y[t].\
\>", "MathCaption"],

Cell[BoxData[
    \(\(f[t_]\ \  := x[t]\^2 + y[t]\^2;\)\)], "Input",
  CellLabel->"In[35]:="],

Cell["\<\
This defines the linear equations along with initial conditions.\
\>", "MathCaption"],

Cell[BoxData[{
    \(\(eqns\  = \ {\(x'\)[t] \[Equal] x[t] - 2\ y[t], \(y'\)[t] \[Equal] 
            x[t] + y[t]};\)\), "\[IndentingNewLine]", 
    \(\(ics\  = \ {x[0]\  \[Equal] \ 1, \ y[0]\  \[Equal] \ 1};\)\)}], "Input",\

  CellLabel->"In[36]:="],

Cell[TextData[{
  "The exact value of f as a function of time can be computed symbolically \
using ",
  StyleBox["DSolve", "MR"],
  "."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(f\_exact[t_]\  = \ 
      First[f[t] /. DSolve[{eqns, \ ics}, {x, y}, t]]\)], "Input",
  CellLabel->"In[38]:="],

Cell[BoxData[
    \(\[ExponentialE]\^\(2\ t\)\ \((Cos[\@2\ t] - \@2\ Sin[\@2\ t])\)\^2 + 
      1\/4\ \[ExponentialE]\^\(2\ t\)\ \((2\ Cos[\@2\ t] + \@2\ Sin[\@2\ \
t])\)\^2\)], "Output",
  CellLabel->"Out[38]="]
}, Open  ]],

Cell["\<\
The exact solution will be compared with solutions computed with and without \
dense output.\
\>", "Text"],

Cell["\<\
A simple way to track the quantity is to create a function which derives it \
from the numerical solution of the differential equation.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(f1[t_]\  = \ 
      First[f[t]\  /. \ 
          NDSolve[{eqns, \ ics}, \ {x, y}, {t, 0, 1}]]\)], "Input",
  CellLabel->"In[39]:="],

Cell[BoxData[
    RowBox[{
      SuperscriptBox[
        RowBox[{
          TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
            False,
            Editable->False], "[", "t", "]"}], "2"], "+", 
      SuperscriptBox[
        RowBox[{
          TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
            False,
            Editable->False], "[", "t", "]"}], "2"]}]], "Output",
  CellLabel->"Out[39]="]
}, Open  ]],

Cell["It can also be computed with dense output.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(f1\_dense[t_]\  = \ 
      First[f[t]\  /. \ 
          NDSolve[{eqns, \ ics}, \ {x, y}, {t, 0, 1}, \ 
            Method \[Rule] {"\<LSODA\>", \ "\<DenseOutput\>" \[Rule] 
                  True}]]\)], "Input",
  CellLabel->"In[40]:="],

Cell[BoxData[
    RowBox[{
      SuperscriptBox[
        RowBox[{
          TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
            False,
            Editable->False], "[", "t", "]"}], "2"], "+", 
      SuperscriptBox[
        RowBox[{
          TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
            False,
            Editable->False], "[", "t", "]"}], "2"]}]], "Output",
  CellLabel->"Out[40]="]
}, Open  ]],

Cell["\<\
This shows a plot showing the error in the two computed solutions. The \
computed solution at the time steps are indicated by black dots.  The default \
output error is shown in gray and the dense output error in black.  \
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(t1\  = \(Cases[
            f1[t], \ \((if_InterpolatingFunction)\)[t] \[Rule] 
              InterpolatingFunctionCoordinates[if], \ Infinity]\)[\([1, 
            1]\)];\)\), "\[IndentingNewLine]", 
    \(\(pode\  = \ 
        Show[Block[{$DisplayFunction\  = \ Identity}, {ListPlot[
                Transpose[{t1, \ f\_exact[t1]\  - \ f1[t1]}], \ 
                PlotStyle \[Rule] PointSize[ .02]], 
              Plot[f\_exact[t] - \ f1[t], {t, 0, 1}, \ 
                PlotStyle \[Rule] RGBColor[ .8,  .8,  .8]], 
              Plot[f\_exact[t] - \ f1\_dense[t], {t, 0, 1}]}], \ 
          PlotRange \[Rule] All];\)\)}], "Input",
  CellLabel->"In[41]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.596169 61969.5 [
[.21429 .58367 -9 -9 ]
[.21429 .58367 9 0 ]
[.40476 .58367 -9 -9 ]
[.40476 .58367 9 0 ]
[.59524 .58367 -9 -9 ]
[.59524 .58367 9 0 ]
[.78571 .58367 -9 -9 ]
[.78571 .58367 9 0 ]
[.97619 .58367 -3 -9 ]
[.97619 .58367 3 0 ]
[.01131 .10041 -39 -6.25 ]
[.01131 .10041 0 6.25 ]
[.01131 .22435 -39 -6.25 ]
[.01131 .22435 0 6.25 ]
[.01131 .34829 -39 -6.25 ]
[.01131 .34829 0 6.25 ]
[.01131 .47223 -39 -6.25 ]
[.01131 .47223 0 6.25 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .59617 m
.21429 .60242 L
s
[(0.2)] .21429 .58367 0 1 Mshowa
.40476 .59617 m
.40476 .60242 L
s
[(0.4)] .40476 .58367 0 1 Mshowa
.59524 .59617 m
.59524 .60242 L
s
[(0.6)] .59524 .58367 0 1 Mshowa
.78571 .59617 m
.78571 .60242 L
s
[(0.8)] .78571 .58367 0 1 Mshowa
.97619 .59617 m
.97619 .60242 L
s
[(1)] .97619 .58367 0 1 Mshowa
.125 Mabswid
.07143 .59617 m
.07143 .59992 L
s
.11905 .59617 m
.11905 .59992 L
s
.16667 .59617 m
.16667 .59992 L
s
.2619 .59617 m
.2619 .59992 L
s
.30952 .59617 m
.30952 .59992 L
s
.35714 .59617 m
.35714 .59992 L
s
.45238 .59617 m
.45238 .59992 L
s
.5 .59617 m
.5 .59992 L
s
.54762 .59617 m
.54762 .59992 L
s
.64286 .59617 m
.64286 .59992 L
s
.69048 .59617 m
.69048 .59992 L
s
.7381 .59617 m
.7381 .59992 L
s
.83333 .59617 m
.83333 .59992 L
s
.88095 .59617 m
.88095 .59992 L
s
.92857 .59617 m
.92857 .59992 L
s
.25 Mabswid
0 .59617 m
1 .59617 L
s
.02381 .10041 m
.03006 .10041 L
s
gsave
.01131 .10041 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-8) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-6) show
grestore
.02381 .22435 m
.03006 .22435 L
s
gsave
.01131 .22435 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-6) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-6) show
grestore
.02381 .34829 m
.03006 .34829 L
s
gsave
.01131 .34829 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-6) show
grestore
.02381 .47223 m
.03006 .47223 L
s
gsave
.01131 .47223 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-6) show
grestore
.125 Mabswid
.02381 .1314 m
.02756 .1314 L
s
.02381 .16238 m
.02756 .16238 L
s
.02381 .19337 m
.02756 .19337 L
s
.02381 .25534 m
.02756 .25534 L
s
.02381 .28632 m
.02756 .28632 L
s
.02381 .31731 m
.02756 .31731 L
s
.02381 .37928 m
.02756 .37928 L
s
.02381 .41026 m
.02756 .41026 L
s
.02381 .44125 m
.02756 .44125 L
s
.02381 .50321 m
.02756 .50321 L
s
.02381 .5342 m
.02756 .5342 L
s
.02381 .56518 m
.02756 .56518 L
s
.02381 .06943 m
.02756 .06943 L
s
.02381 .03844 m
.02756 .03844 L
s
.02381 .00746 m
.02756 .00746 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.02 w
.02381 .59617 Mdot
.02388 .59748 Mdot
.02395 .59878 Mdot
.0262 .60029 Mdot
.02845 .6018 Mdot
.03069 .60332 Mdot
.03788 .59584 Mdot
.04506 .596 Mdot
.05224 .59617 Mdot
.05943 .59634 Mdot
.07379 .59631 Mdot
.08816 .59578 Mdot
.10252 .59572 Mdot
.11689 .59567 Mdot
.13126 .59562 Mdot
.16448 .5948 Mdot
.1977 .59491 Mdot
.23093 .5946 Mdot
.26415 .59413 Mdot
.29737 .59367 Mdot
.3306 .59314 Mdot
.38251 .59032 Mdot
.43442 .59067 Mdot
.48632 .5901 Mdot
.53823 .58833 Mdot
.59014 .58633 Mdot
.64205 .58378 Mdot
.69396 .58065 Mdot
.75066 .57551 Mdot
.80736 .56934 Mdot
.86405 .56154 Mdot
.92012 .5523 Mdot
.97619 .54098 Mdot
.8 .8 .8 r
.5 Mabswid
.02381 .59617 m
.02499 .59945 L
.02605 .60027 L
.02846 .6018 L
.02952 .60251 L
.03068 .60332 L
.03178 .60286 L
.03279 .60179 L
.03397 .60006 L
.03525 .5981 L
.03646 .5966 L
.03756 .59588 L
.03879 .59584 L
.03997 .59586 L
.04123 .5959 L
.04262 .59595 L
.04326 .59597 L
.04396 .59599 L
.04523 .596 L
.04646 .59601 L
.04759 .59604 L
.0529 .59617 L
.05406 .59619 L
.0553 .59623 L
.05754 .59631 L
.05876 .59634 L
.05988 .59634 L
.0612 .59631 L
.06244 .59626 L
.06361 .59622 L
.06486 .59618 L
.06558 .59616 L
.06624 .59615 L
.06753 .59615 L
.06878 .59618 L
.0695 .59619 L
.07016 .59622 L
.07158 .59627 L
.07228 .59629 L
.07293 .5963 L
.07411 .59631 L
.07538 .59627 L
.07647 .5962 L
.07762 .59611 L
.07896 .59601 L
.08023 .59591 L
.08138 .59585 L
.08263 .5958 L
.08328 .59578 L
.08398 .59577 L
Mistroke
.08523 .59576 L
.08599 .59576 L
.0867 .59577 L
.08734 .59577 L
.08803 .59578 L
.08929 .59576 L
.09046 .59572 L
.09183 .59566 L
.09308 .59561 L
.09378 .59559 L
.09445 .59558 L
.09516 .59557 L
.09592 .59557 L
.09659 .59557 L
.0973 .59558 L
.09857 .59562 L
.09994 .59567 L
.10067 .59569 L
.10143 .59571 L
.10228 .59572 L
.10305 .59572 L
.10378 .5957 L
.10458 .59568 L
.10713 .59557 L
.10782 .59555 L
.10856 .59553 L
.10988 .59551 L
.11111 .59552 L
.11226 .59554 L
.11353 .59558 L
.11488 .59563 L
.11608 .59566 L
.11739 .59567 L
.11863 .59563 L
.11977 .59559 L
.12081 .59554 L
.12191 .5955 L
.12312 .59547 L
.12423 .59545 L
.12542 .59546 L
.12672 .59549 L
.12793 .59553 L
.12906 .59557 L
.12975 .59559 L
.13041 .59561 L
.13163 .5956 L
.13296 .5954 L
.1337 .59519 L
.13439 .59495 L
.1396 .59235 L
Mistroke
.14247 .59093 L
.14388 .59037 L
.14519 .58996 L
.14636 .58971 L
.14761 .58956 L
.14832 .58953 L
.14899 .58953 L
.15025 .58965 L
.15141 .58987 L
.15247 .59016 L
.15487 .59107 L
.15713 .59215 L
.15956 .59337 L
.16073 .5939 L
.16185 .59432 L
.16285 .5946 L
.16393 .59478 L
.16514 .59477 L
.16642 .59452 L
.16751 .59417 L
.16872 .59366 L
.17397 .59092 L
.1766 .58976 L
.17732 .58951 L
.17808 .5893 L
.17873 .58914 L
.17943 .58902 L
.1807 .58889 L
.18188 .5889 L
.18261 .58896 L
.18329 .58906 L
.18456 .58936 L
.18572 .58973 L
.18696 .59024 L
.1892 .59135 L
.19164 .59272 L
.19429 .59408 L
.19499 .59436 L
.19573 .59461 L
.19643 .59478 L
.19706 .59488 L
.19771 .59491 L
.19841 .59487 L
.19968 .5946 L
.20092 .59414 L
.20224 .59352 L
.20459 .59218 L
.20885 .58981 L
.21007 .58928 L
.21123 .58888 L
Mistroke
.21227 .58861 L
.2134 .58842 L
.21466 .58835 L
.2153 .58837 L
.216 .58843 L
.21723 .58866 L
.2184 .58899 L
.21957 .58943 L
.22064 .58991 L
.22305 .59122 L
.2255 .59266 L
.22781 .59385 L
.22909 .59432 L
.22981 .5945 L
.23047 .59459 L
.23165 .59455 L
.23289 .59426 L
.23398 .59384 L
.23514 .59326 L
.23724 .592 L
.23973 .59041 L
.24203 .58909 L
.24329 .58848 L
.24469 .58796 L
.24542 .58776 L
.2461 .58761 L
.24682 .58751 L
.24759 .58745 L
.24823 .58745 L
.24893 .5875 L
.24966 .58759 L
.25034 .58773 L
.25156 .58809 L
.25286 .58861 L
.25766 .59138 L
.25893 .59217 L
.26029 .59295 L
.26155 .59355 L
.26275 .59395 L
.26338 .59408 L
.26407 .59413 L
.26528 .594 L
.26598 .59381 L
.26673 .59352 L
.26807 .59285 L
.27298 .58957 L
.27442 .58864 L
.27598 .58777 L
.27745 .58712 L
.27815 .58687 L
Mistroke
.27881 .58669 L
.28003 .58646 L
.28068 .58641 L
.28137 .5864 L
.2821 .58644 L
.28279 .58654 L
.28408 .58686 L
.28524 .58728 L
.28652 .58789 L
.2891 .58945 L
.29152 .59111 L
.29374 .59251 L
.29501 .59314 L
.29565 .59338 L
.29635 .59356 L
.29705 .59366 L
.2978 .59365 L
.29851 .59352 L
.29915 .59333 L
.30041 .59278 L
.30177 .59197 L
.30423 .59018 L
.30654 .58844 L
.30776 .58759 L
.30906 .5868 L
.31019 .58621 L
.31144 .58571 L
.31258 .5854 L
.31364 .58524 L
.31429 .58521 L
.31497 .58523 L
.31618 .5854 L
.31749 .58577 L
.31823 .58607 L
.31893 .58639 L
.32375 .5896 L
.32507 .59059 L
.3265 .59159 L
.32776 .59234 L
.32845 .59266 L
.3291 .5929 L
.32984 .59308 L
.33066 .59314 L
.33136 .59297 L
.3321 .59248 L
.33283 .59173 L
.33353 .59078 L
.33483 .58846 L
.3374 .58224 L
.33978 .57517 L
Mistroke
.34519 .55786 L
.34767 .55065 L
.35035 .54416 L
.35158 .54176 L
.35288 .53967 L
.35399 .53829 L
.3552 .5372 L
.35639 .5366 L
.35747 .53645 L
.35872 .53676 L
.3599 .53752 L
.36099 .53862 L
.362 .53997 L
.36428 .54414 L
.36934 .5578 L
.37478 .57549 L
.37728 .5827 L
.37865 .58591 L
.37992 .58823 L
.38101 .58959 L
.38217 .59028 L
.38346 .59 L
.38463 .58882 L
.3858 .58691 L
.38703 .58419 L
.38924 .57792 L
.39424 .56011 L
.39861 .54448 L
.40102 .5372 L
.4033 .5317 L
.40457 .52936 L
.40524 .52834 L
.40595 .52743 L
.40721 .52628 L
.40837 .52575 L
.40966 .52576 L
.41086 .52635 L
.41195 .52735 L
.41312 .52894 L
.41442 .53129 L
.41585 .53454 L
.41874 .54298 L
.42391 .56224 L
.42643 .57205 L
.42917 .58154 L
.43042 .58508 L
.43111 .58673 L
.43175 .58805 L
.43289 .58977 L
.43414 .59064 L
Mistroke
.43534 .59031 L
.43649 .58898 L
.43751 .58706 L
.43863 .58427 L
.44113 .57598 L
.44346 .56655 L
.44867 .54364 L
.45123 .53334 L
.45359 .5253 L
.45488 .52164 L
.45631 .51834 L
.45774 .51585 L
.45852 .51485 L
.45925 .51416 L
.46055 .51354 L
.46173 .51364 L
.46239 .51396 L
.4631 .51454 L
.46439 .51618 L
.46561 .51839 L
.46672 .52097 L
.46923 .52854 L
.47378 .54708 L
.47873 .56982 L
.48135 .58028 L
.48281 .58489 L
.48351 .58665 L
.48415 .58799 L
.48545 .58975 L
.48614 .59008 L
.48687 .58994 L
.4881 .58859 L
.48879 .58727 L
.48944 .5857 L
.49182 .5777 L
.49437 .56619 L
.49996 .53689 L
.50268 .52343 L
.50515 .51298 L
.50636 .50866 L
.50767 .5047 L
.50881 .50192 L
.51005 .49961 L
.51137 .49803 L
.51209 .49757 L
.51277 .49739 L
.51397 .49768 L
.51527 .49889 L
.51601 .49998 L
.51669 .50124 L
Mistroke
.51823 .50499 L
.5196 .50927 L
.52104 .51472 L
.5263 .5405 L
.5309 .56558 L
.53332 .5769 L
.53464 .58185 L
.53586 .58537 L
.53652 .58675 L
.53722 .58777 L
.53799 .5883 L
.5387 .58819 L
.53934 .5876 L
.54004 .58646 L
.5413 .58322 L
.54259 .57855 L
.54395 .57237 L
.5464 .55896 L
.55125 .52845 L
.55585 .50172 L
.55839 .49027 L
.55982 .48523 L
.56115 .4816 L
.56239 .4792 L
.5635 .47785 L
.56479 .47728 L
.566 .47774 L
.56706 .47894 L
.56823 .48109 L
.56945 .48426 L
.57059 .48802 L
.57263 .49657 L
.57481 .50796 L
.57942 .53704 L
.58432 .56787 L
.58553 .57413 L
.58684 .57972 L
.58759 .58227 L
.58828 .58411 L
.58897 .58544 L
.58963 .58616 L
.59084 .58597 L
.59197 .58402 L
.59321 .58018 L
.59453 .57442 L
.59752 .55652 L
.60034 .53596 L
.60562 .49638 L
.60807 .48047 L
.61066 .46696 L
Mistroke
.61172 .46259 L
.61286 .45877 L
.61412 .45567 L
.61529 .45384 L
.61599 .45325 L
.61664 .45305 L
.61735 .4532 L
.6181 .4538 L
.61937 .45581 L
.62009 .4575 L
.62076 .45945 L
.62317 .46904 L
.62576 .48357 L
.63077 .52032 L
.63353 .54215 L
.63614 .56108 L
.63731 .56844 L
.63856 .57501 L
.63922 .57783 L
.63993 .58037 L
.6412 .58321 L
.64234 .58369 L
.64357 .58183 L
.64473 .57804 L
.64579 .57308 L
.64827 .55712 L
.65096 .53492 L
.65587 .49021 L
.65864 .4668 L
.66159 .44615 L
.66302 .43825 L
.66437 .4323 L
.66558 .42827 L
.66689 .42539 L
.66814 .42411 L
.66927 .42422 L
.67058 .42584 L
.67182 .42886 L
.67297 .43288 L
.67403 .43762 L
.67642 .45171 L
.68149 .49345 L
.68408 .51801 L
.68687 .54368 L
.68818 .55459 L
.68957 .56463 L
.69085 .57217 L
.69204 .57727 L
.69316 .58005 L
.69436 .58044 L
Mistroke
.69504 .5792 L
.69569 .57709 L
.69689 .57087 L
.69753 .56648 L
.69823 .56087 L
.69965 .54733 L
.7022 .51728 L
.70794 .43637 L
.71045 .40153 L
.71317 .36745 L
.71557 .34245 L
.71689 .33123 L
.71813 .32248 L
.71923 .31624 L
.72042 .31123 L
.72174 .30784 L
.72296 .30678 L
.72399 .30745 L
.72513 .30986 L
.72632 .31423 L
.72742 .31997 L
.72873 .32873 L
.73014 .34057 L
.73272 .36767 L
.73753 .43282 L
.74022 .47302 L
.74311 .51455 L
.74439 .53114 L
.74578 .54714 L
.74712 .55976 L
.74833 .5684 L
.74957 .57391 L
.75089 .57542 L
.75211 .5724 L
.75326 .56599 L
.75439 .55678 L
.75543 .54608 L
.75778 .51557 L
.76274 .43393 L
.76818 .34218 L
.77073 .30576 L
.77214 .28869 L
.77347 .2748 L
.77471 .26405 L
.77605 .25491 L
.77728 .24901 L
.77841 .24566 L
.77969 .24437 L
.7809 .24561 L
.78159 .24739 L
Mistroke
.78222 .24971 L
.78292 .25299 L
.78365 .25728 L
.78502 .2676 L
.78631 .27974 L
.78921 .31549 L
.79427 .39788 L
.79974 .49485 L
.80225 .53224 L
.80352 .54732 L
.80425 .5545 L
.80491 .55996 L
.8061 .5668 L
.80738 .56934 L
.80856 .56672 L
.80968 .56008 L
.81085 .54931 L
.81192 .53664 L
.8143 .50011 L
.81858 .41668 L
.8232 .32057 L
.82583 .27119 L
.82827 .23199 L
.83064 .20227 L
.83196 .18967 L
.83317 .18093 L
.83444 .17467 L
.83516 .1725 L
.83582 .17139 L
.83704 .17161 L
.83833 .17502 L
.83942 .18045 L
.8406 .18885 L
.84171 .19914 L
.84272 .21049 L
.84757 .28673 L
.8528 .39567 L
.85567 .4568 L
.85837 .50769 L
.8596 .52682 L
.8609 .54326 L
.86158 .55002 L
.86232 .5558 L
.86363 .56122 L
.86482 .56023 L
.8661 .553 L
.86731 .54106 L
.86841 .52642 L
.87083 .48402 L
.87307 .43587 L
Mistroke
.87815 .31418 L
.88291 .20912 L
.88524 .16803 L
.88738 .13871 L
.8885 .12697 L
.88968 .11747 L
.89086 .11098 L
.89215 .10748 L
.89347 .10786 L
.89421 .10981 L
.89488 .11267 L
.89619 .12115 L
.89742 .13263 L
.89872 .14814 L
.90009 .16822 L
.90254 .21272 L
.9081 .34131 L
.91293 .45849 L
.91532 .50606 L
.91648 .52448 L
.91753 .53768 L
.91875 .54811 L
.92006 .5523 L
.92074 .55127 L
.92148 .54767 L
.92218 .54208 L
.92282 .5352 L
.92394 .51951 L
.92517 .49764 L
.92766 .44185 L
.93258 .30599 L
.93724 .17706 L
.94142 .08424 L
.94257 .06471 L
.94378 .04725 L
.94492 .03413 L
.94597 .02499 L
.94681 .0197 L
.94769 .01616 L
.94858 .01472 L
.94955 .01558 L
.95039 .01839 L
.95129 .02351 L
.95214 .03031 L
.95291 .03812 L
.9548 .06366 L
.95685 .10084 L
.96056 .1893 L
.97619 .54098 L
Mfstroke
0 g
.02381 .59617 m
.02846 .59884 L
.02952 .60209 L
.03068 .60332 L
.03178 .60286 L
.03279 .60179 L
.03399 .60003 L
.03527 .59807 L
.03635 .5967 L
.03754 .59589 L
.0388 .59584 L
.03951 .59585 L
.04016 .59587 L
.04142 .59591 L
.04262 .59595 L
.04387 .59598 L
.045 .596 L
.04622 .59601 L
.04753 .59604 L
.04876 .59608 L
.04988 .59612 L
.05096 .59615 L
.0521 .59617 L
.05488 .59624 L
.05554 .59625 L
.05624 .59627 L
.05751 .59631 L
.05863 .59633 L
.05987 .59634 L
.06244 .59637 L
.06369 .59638 L
.06439 .59638 L
.06503 .59639 L
.06621 .59639 L
.06746 .59639 L
.0681 .59638 L
.06879 .59637 L
.07003 .59636 L
.07283 .59632 L
.07409 .59631 L
.07479 .59631 L
.07545 .5963 L
.07665 .59627 L
.07791 .59623 L
.08011 .59613 L
.08253 .59599 L
.08388 .59592 L
.08535 .59585 L
.08602 .59582 L
.08674 .5958 L
Mistroke
.08802 .59578 L
.08886 .59578 L
.08964 .59577 L
.09111 .59577 L
.09246 .59577 L
.09392 .59576 L
.10458 .59572 L
.12456 .59564 L
.12941 .59562 L
.13069 .59562 L
.13137 .59562 L
.13209 .59561 L
.1334 .59561 L
.13461 .59559 L
.13603 .59558 L
.1376 .59555 L
.14032 .59549 L
.14564 .59533 L
.15508 .59499 L
.15738 .59492 L
.15983 .59486 L
.16124 .59483 L
.16252 .59481 L
.16381 .5948 L
.16502 .5948 L
.16565 .5948 L
.16634 .5948 L
.16707 .5948 L
.16774 .5948 L
.16895 .5948 L
.17025 .59481 L
.17177 .59481 L
.17321 .59482 L
.17591 .59483 L
.18065 .59486 L
.18317 .59488 L
.18589 .59489 L
.18848 .59491 L
.18962 .59491 L
.19086 .59491 L
.19204 .59492 L
.19313 .59492 L
.19429 .59492 L
.19553 .59492 L
.19622 .59492 L
.19685 .59492 L
.19755 .59491 L
.19829 .59491 L
.2009 .5949 L
.20322 .59489 L
Mistroke
.20578 .59488 L
.21036 .59484 L
.21528 .59479 L
.22419 .59467 L
.26543 .59412 L
.28604 .59384 L
.29121 .59375 L
.29607 .59368 L
.30081 .59363 L
.30288 .59361 L
.30516 .59359 L
.30948 .59353 L
.31405 .59344 L
.32366 .59324 L
.32896 .59316 L
.33156 .59313 L
.33271 .59312 L
.33397 .5931 L
.33616 .59306 L
.33848 .593 L
.34337 .59282 L
.34822 .59257 L
.35345 .59223 L
.3629 .59151 L
.37289 .59073 L
.37563 .59057 L
.37713 .59049 L
.37855 .59042 L
.37988 .59038 L
.38057 .59036 L
.38131 .59034 L
.38262 .59032 L
.38382 .5903 L
.3862 .59028 L
.38751 .59027 L
.38875 .59027 L
.38991 .59026 L
.39117 .59026 L
.39231 .59026 L
.39337 .59025 L
.39403 .59026 L
.39475 .59026 L
.39604 .59026 L
.3973 .59026 L
.39846 .59027 L
.40109 .59029 L
.40387 .59032 L
.40909 .5904 L
.41394 .59048 L
.41923 .59058 L
Mistroke
.42202 .59062 L
.42355 .59065 L
.42498 .59066 L
.42633 .59068 L
.42757 .59069 L
.42828 .59069 L
.42895 .59069 L
.42966 .5907 L
.43043 .5907 L
.43169 .59069 L
.4324 .59069 L
.43304 .59068 L
.43429 .59067 L
.43547 .59065 L
.44539 .59054 L
.45431 .59046 L
.46396 .59038 L
.47346 .5903 L
.47885 .59023 L
.48389 .59015 L
.49405 .58992 L
.50495 .58959 L
.5463 .5881 L
.58613 .58648 L
.62445 .58473 L
.66522 .58257 L
.68546 .58117 L
.69538 .58057 L
.70447 .58001 L
.71332 .57932 L
.72263 .57843 L
.74221 .57626 L
.78031 .57269 L
.82086 .56798 L
.85989 .56206 L
.90137 .5557 L
.94134 .54877 L
.97619 .54098 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[41]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00:P3oool00`000000oooo0?ooo`3c0?ooo`00:P3oool200000?@0oooo000Z0?ooo`030000
003oool0oooo0?<0oooo000Z0?ooo`030000003oool0oooo0>D0oooo00<0c<c<0?ooo`3oool02`3o
ool002X0oooo00<000000?ooo`3oool0i03oool20<c<c0d0oooo000Z0?ooo`030000003oool0oooo
0>@0oooo0P3<c<`=0?ooo`00:P3oool00`000000oooo0?ooo`3T0?ooo`80c<c<3@3oool002X0oooo
00<000000?ooo`3oool0i03oool20<c<c0d0oooo000Z0?ooo`030000003oool0oooo0>@0oooo0P3<
c<`=0?ooo`00:P3oool200000>D0oooo00<0c<c<0?ooo`3<c<`0303oool002X0oooo00<000000?oo
o`3oool0h`3oool0103<c<`0oooo0?ooo`3<c<`<0?ooo`00:P3oool00`000000oooo0?ooo`3S0?oo
o`040<c<c03oool0oooo0<c<c0`0oooo000Z0?ooo`030000003oool0oooo0><0oooo00@0c<c<0?oo
o`3oool0c<c<303oool002X0oooo00<000000?ooo`3oool0h`3oool0103<c<`0oooo0?ooo`3<c<`<
0?ooo`00:P3oool00`000000oooo0?ooo`3S0?ooo`040<c<c03oool0oooo0<c<c0`0oooo000Z0?oo
o`030000003oool0oooo0><0oooo00@0c<c<0?ooo`3oool0c<c<303oool002X0oooo00<000000?oo
o`3oool0h`3oool0103<c<`0oooo0?ooo`3<c<`<0?ooo`00:P3oool200000>@0oooo00@0c<c<0?oo
o`3oool0c<c<303oool002X0oooo00<000000?ooo`3oool0h`3oool0103<c<`0oooo0?ooo`3<c<`<
0?ooo`00:P3oool00`000000oooo0?ooo`3S0?ooo`040<c<c03oool0oooo0<c<c0`0oooo000Z0?oo
o`030000003oool0oooo0><0oooo00@0c<c<0?ooo`3oool0c<c<303oool000P0oooo0P0000030?oo
o`050000003oool0oooo0?ooo`000000103oool500000080oooo0P00000;0?ooo`030000003oool0
oooo0><0oooo00D0c<c<0?ooo`3oool0oooo0<c<c00;0?ooo`001`3oool010000000oooo0?ooo`00
00030?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool0h`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00/0
oooo00070?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool01P3oool01@000000
oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`3S0?oo
o`050<c<c03oool0oooo0?ooo`3<c<`02`3oool000030?ooo`0000000000008000000`3oool20000
00@0oooo00<000000?ooo`0000001`3oool01@000000oooo0?ooo`3oool000000080oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`3S0?ooo`050<c<c03oool0oooo0?ooo`3<c<`02`3o
ool000L0oooo00@000000?ooo`3oool000000P3oool01@000000oooo0?ooo`3oool0000000H0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00P0oooo0P00003T0?ooo`05
0<c<c03oool0oooo0?ooo`3<c<`02`3oool000P0oooo0P00000=0?ooo`800000103oool2000000/0
oooo00<000000?ooo`3oool0eP3oool00`3<c<`0oooo0?ooo`0:0?ooo`050<c<c03oool0oooo0?oo
o`3<c<`02`3oool002L0oooo1000003H0?ooo`80c<c<2`3oool01@3<c<`0oooo0?ooo`3oool0c<c<
00/0oooo000W0?ooo`040000003oool0000000000=P0oooo0P3<c<`;0?ooo`050<c<c03oool0oooo
0?ooo`3<c<`02`3oool00280oooo0`0000020?ooo`@00000f03oool20<c<c0/0oooo00D0c<c<0?oo
o`3oool0oooo0<c<c00;0?ooo`009`3oool010000000oooo0?ooo`00003H0?ooo`80c<c<2`3oool0
1@3<c<`0oooo0?ooo`3oool0c<c<00/0oooo000X0?ooo`<00000f03oool20<c<c0/0oooo00D0c<c<
0?ooo`3oool0oooo0<c<c00;0?ooo`00:P3oool200000=L0oooo00<0c<c<0?ooo`3<c<`02P3oool0
1@3<c<`0oooo0?ooo`3oool0c<c<00/0oooo000Z0?ooo`030000003oool0oooo0=D0oooo00@0c<c<
0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00T0oooo000Z
0?ooo`030000003oool0oooo0=D0oooo00@0c<c<0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?oo
o`020?ooo`030<c<c03oool0oooo00T0oooo000Z0?ooo`030000003oool0oooo0=D0oooo00@0c<c<
0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00T0oooo000Z
0?ooo`030000003oool0oooo0=D0oooo00@0c<c<0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?oo
o`020?ooo`030<c<c03oool0oooo00T0oooo000Z0?ooo`030000003oool0oooo0=D0oooo00@0c<c<
0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00P0oooo000Z
0?ooo`030000003oool0oooo0=D0oooo00@0c<c<0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?oo
o`030?ooo`030<c<c03oool0oooo00P0oooo000Z0?ooo`030000003oool0oooo0=D0oooo00@0c<c<
0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00P0oooo000Z
0?ooo`800000eP3oool0103<c<`0oooo0?ooo`3<c<`90?ooo`030<c<c03oool0oooo00<0oooo00<0
c<c<0?ooo`3oool0203oool002X0oooo00<000000?ooo`3oool0e@3oool0103<c<`0oooo0?ooo`3<
c<`90?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0203oool002X0oooo00<00000
0?ooo`3oool0e@3oool0103<c<`0oooo0?ooo`3<c<`90?ooo`030<c<c03oool0oooo00<0oooo00<0
c<c<0?ooo`3oool0203oool002X0oooo00<000000?ooo`3oool0b03oool20<c<c0/0oooo00@0c<c<
0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00P0oooo000Z
0?ooo`030000003oool0oooo0<P0oooo0P3<c<`;0?ooo`040<c<c03oool0oooo0<c<c0T0oooo00<0
c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`080?ooo`00:P3oool00`000000oooo0?ooo`38
0?ooo`80c<c<2`3oool0103<c<`0oooo0?ooo`3<c<`90?ooo`030<c<c03oool0oooo00<0oooo00<0
c<c<0?ooo`3oool0203oool002X0oooo00<000000?ooo`3oool0a`3oool00`3<c<`0oooo0<c<c00;
0?ooo`040<c<c03oool0oooo0<c<c0T0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?oo
o`080?ooo`00:P3oool00`000000oooo0?ooo`370?ooo`040<c<c03oool0oooo0<c<c0T0oooo00D0
c<c<0?ooo`3oool0oooo0<c<c0090?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0
203oool002X0oooo0P0000380?ooo`040<c<c03oool0oooo0<c<c0T0oooo00<0c<c<0?ooo`3oool0
0P3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0
203oool002X0oooo00<000000?ooo`3oool0a`3oool0103<c<`0oooo0?ooo`3<c<`90?ooo`030<c<
c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<
c03oool0oooo00P0oooo000Z0?ooo`030000003oool0oooo0<L0oooo00@0c<c<0?ooo`3oool0c<c<
2@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool0
0`3oool00`3<c<`0oooo0?ooo`080?ooo`00:P3oool00`000000oooo0?ooo`370?ooo`040<c<c03o
ool0oooo0<c<c0T0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<
c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0203oool002X0oooo00<000000?ooo`3oool0a`3o
ool0103<c<`0oooo0?ooo`3<c<`90?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool0
1P3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00P0oooo00070?ooo`<000000`3o
ool01@000000oooo0?ooo`3oool0000000@0oooo1@0000020?ooo`8000002`3oool00`000000oooo
0?ooo`370?ooo`040<c<c03oool0oooo0<c<c0T0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0
oooo0?ooo`060?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0203oool000L0oooo
00@000000?ooo`3oool000000`3oool00`000000oooo000000070?ooo`050000003oool0oooo0?oo
o`0000000P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo0<H0oooo00D0c<c<0?oo
o`3oool0oooo0<c<c0090?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool0
0`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00P0oooo00070?ooo`040000003oool0oooo
000000@0oooo00<000000?ooo`3oool01P3oool01@000000oooo0?ooo`3oool000000080oooo00<0
00000?ooo`3oool0203oool00`000000oooo0?ooo`360?ooo`050<c<c03oool0oooo0?ooo`3<c<`0
2@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool0
0`3oool00`3<c<`0oooo0?ooo`080?ooo`0000<0oooo0000000000000P0000020?ooo`<00000103o
ool00`000000oooo000000070?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo
0?ooo`080?ooo`800000a`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00T0oooo00<0c<c<0?ooo`3o
ool00P3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3o
ool0203oool000P0oooo00<000000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool0000000H0
oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool0aP3oool01@3<c<`0oooo0?ooo`3oool0c<c<00T0oooo00<0c<c<0?ooo`3oool00P3oool0
0`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0203oool0
00P0oooo0`00000<0?ooo`800000103oool2000000P0oooo100000380?ooo`050<c<c03oool0oooo
0?ooo`3<c<`02@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<
0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`080?ooo`009`3oool010000000oooo000000000038
0?ooo`050<c<c03oool0oooo0?ooo`3<c<`02@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03o
ool0oooo00H0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`070?ooo`008P3oool3
00000080oooo100000380?ooo`050<c<c03oool0oooo0?ooo`3<c<`02@3oool00`3<c<`0oooo0?oo
o`020?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?oo
o`070?ooo`009`3oool010000000oooo0?ooo`00002l0?ooo`80c<c<2P3oool01@3<c<`0oooo0?oo
o`3oool0c<c<00T0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<
c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool01`3oool002P0oooo0`00002l0?ooo`80c<c<2P3o
ool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool00P3o
ool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool01`3o
ool002X0oooo00<000000?ooo`3oool0^@3oool00`3<c<`0oooo0<c<c00:0?ooo`030<c<c03oool0
oooo0080oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0
oooo00D0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`070?ooo`00:P3oool20000
0;X0oooo00<0c<c<0?ooo`3<c<`02P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo
00H0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo
00D0oooo00<0c<c<0?ooo`3oool01`3oool002X0oooo00<000000?ooo`3oool0^@3oool00`3<c<`0
oooo0<c<c00:0?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0
oooo0?ooo`030?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0
oooo0?ooo`070?ooo`00:P3oool00`000000oooo0?ooo`2i0?ooo`040<c<c03oool0oooo0<c<c0T0
oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0
oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00L0
oooo000Z0?ooo`030000003oool0oooo0;T0oooo00@0c<c<0?ooo`3oool0c<c<2@3oool00`3<c<`0
oooo0?ooo`020?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0
oooo0?ooo`050?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool01`3oool002X0oooo
00<000000?ooo`3oool0^@3oool0103<c<`0oooo0?ooo`3<c<`90?ooo`030<c<c03oool0oooo0080
oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00D0
oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`070?ooo`00:P3oool00`000000oooo
0?ooo`2i0?ooo`040<c<c03oool0oooo0<c<c0T0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0
oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0
oooo0?ooo`050?ooo`030<c<c03oool0oooo00L0oooo000Z0?ooo`030000003oool0oooo0;P0oooo
00D0c<c<0?ooo`3oool0oooo0<c<c0090?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3o
ool01@3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3o
ool01@3oool00`3<c<`0oooo0?ooo`070?ooo`00:P3oool200000;T0oooo00D0c<c<0?ooo`3oool0
oooo0<c<c0090?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0
oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0
oooo0?ooo`070?ooo`00:P3oool00`000000oooo0?ooo`2h0?ooo`050<c<c03oool0oooo0?ooo`3<
c<`02@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3o
ool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3o
ool01`3oool002X0oooo00<000000?ooo`3oool0^03oool01@3<c<`0oooo0?ooo`3oool0c<c<00T0
oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00@0
oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00L0
oooo000Z0?ooo`030000003oool0oooo0;P0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0080?ooo`03
0<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`040?ooo`03
0<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`070?ooo`00
:P3oool00`000000oooo0?ooo`2h0?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool0
1@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool0
103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool0
1`3oool002X0oooo00<000000?ooo`3oool0[03oool00`3<c<`0oooo0?ooo`090?ooo`030<c<c03o
ool0oooo0080oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03o
ool0oooo00D0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03o
ool0oooo00D0oooo00<0c<c<0?ooo`3oool01`3oool002X0oooo00<000000?ooo`3oool0[03oool2
0<c<c0X0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0
oooo00<0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0
oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`070?ooo`00:P3oool00`00
0000oooo0?ooo`2[0?ooo`030<c<c03oool0c<c<00X0oooo00<0c<c<0?ooo`3oool00P3oool00`3<
c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01@3oool00`3<
c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool00`3<
c<`0oooo0?ooo`070?ooo`00:P3oool200000:`0oooo00<0c<c<0?ooo`3<c<`02P3oool00`3<c<`0
oooo0?ooo`020?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0
oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0
oooo0?ooo`050?ooo`030<c<c03oool0oooo00L0oooo000Z0?ooo`030000003oool0oooo0:/0oooo
00<0c<c<0?ooo`3<c<`02P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00D0oooo
00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo
00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00L0oooo
000Z0?ooo`030000003oool0oooo0:/0oooo00<0c<c<0?ooo`3<c<`02@3oool00`3<c<`0oooo0?oo
o`030?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?oo
o`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?oo
o`060?ooo`030<c<c03oool0oooo00H0oooo000Z0?ooo`030000003oool0oooo0:X0oooo00@0c<c<
0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00D0oooo00<0
c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0
c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo00H0oooo000Z
0?ooo`030000003oool0oooo0:X0oooo00@0c<c<0?ooo`3oool0c<c<2@3oool00`3<c<`0oooo0?oo
o`030?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?oo
o`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?oo
o`060?ooo`030<c<c03oool0oooo00H0oooo00090?ooo`<0000000<0oooo0000003oool00P3oool0
0`000000oooo0?ooo`020?ooo`D000000P3oool2000000/0oooo00<000000?ooo`3oool0ZP3oool0
1@3<c<`0oooo0?ooo`3oool0c<c<00P0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?oo
o`050?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?oo
o`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?oo
o`060?ooo`002P3oool01`000000oooo0?ooo`3oool000000?ooo`0000001`3oool01@000000oooo
0?ooo`3oool000000080oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`2Z0?ooo`05
0<c<c03oool0oooo0?ooo`3<c<`0203oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo
00D0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo
00@0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`070?ooo`030<c<c03oool0oooo
00H0oooo00070?ooo`D000000`3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?oo
o`0000000P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo0:X0oooo00D0c<c<0?oo
o`3oool0oooo0<c<c0080?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01@3oool0
0`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool0
0`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00L0oooo00<0c<c<0?ooo`3oool01P3oool0
00030?ooo`0000000000008000000P3oool010000000oooo0?ooo`0000030?ooo`030000003oool0
000000L0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00P0oooo0P00
002[0?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool00`3<c<`0oooo0?ooo`030?ooo`030<c<
c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<
c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`070?ooo`030<c<
c03oool0oooo00H0oooo00080?ooo`030000003oool000000080oooo00D000000?ooo`3oool0oooo
000000060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo0:X0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0080?ooo`030<c<c03oool0oooo
00<0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo
00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo
00L0oooo00<0c<c<0?ooo`3oool01P3oool000T0oooo0P00000<0?ooo`800000103oool2000000P0
oooo1000002/0?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool00`3<c<`0oooo0?ooo`030?oo
o`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?oo
o`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`070?oo
o`030<c<c03oool0oooo00H0oooo000W0?ooo`040000003oool0000000000:`0oooo00D0c<c<0?oo
o`3oool0oooo0<c<c0080?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool0
0`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool0
0`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00L0oooo00<0c<c<0?ooo`3oool01P3oool0
0280oooo0`0000020?ooo`@00000[03oool01@3<c<`0oooo0?ooo`3oool0c<c<00P0oooo00<0c<c<
0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<
0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<
0?ooo`3oool01`3oool00`3<c<`0oooo0?ooo`060?ooo`009`3oool010000000oooo0?ooo`00002/
0?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03o
ool0oooo00<0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03o
ool0oooo00@0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`070?ooo`030<c<c03o
ool0oooo00H0oooo000X0?ooo`<00000[03oool01@3<c<`0oooo0?ooo`3oool0c<c<00P0oooo00<0
c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00H0oooo00<0
c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo0080oooo00<0
c<c<0?ooo`3oool01`3oool00`3<c<`0oooo0?ooo`060?ooo`00:P3oool00`000000oooo0?ooo`2Z
0?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03o
ool0oooo00<0oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03o
ool0oooo00H0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`070?ooo`030<c<c03o
ool0oooo00H0oooo000Z0?ooo`800000Z`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00P0oooo00<0
c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00H0oooo00<0
c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0
c<c<0?ooo`3oool01`3oool00`3<c<`0oooo0?ooo`060?ooo`00:P3oool00`000000oooo0?ooo`2Y
0?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`04
0?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`02
0?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`07
0?ooo`030<c<c03oool0oooo00H0oooo000Z0?ooo`030000003oool0oooo0:T0oooo00<0c<c<0?oo
o`3oool00`3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?oo
o`3oool00`3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?oo
o`3oool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00L0oooo00<0c<c<0?oo
o`3oool01P3oool002X0oooo00<000000?ooo`3oool0Z@3oool00`3<c<`0oooo0?ooo`030?ooo`03
0<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`03
0<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`060?ooo`03
0<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool00`3<c<`0oooo0?ooo`060?ooo`00
:P3oool00`000000oooo0?ooo`2Y0?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0
103oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0
1P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool0
0P3oool00`3<c<`0oooo0?ooo`070?ooo`030<c<c03oool0oooo00H0oooo000Z0?ooo`030000003o
ool0oooo0:T0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03o
ool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03o
ool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03o
ool0oooo00L0oooo00<0c<c<0?ooo`3oool01P3oool002X0oooo00<000000?ooo`3oool0Z@3oool0
0`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool0
0`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool00P3oool0
0`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool0
0`3<c<`0oooo0?ooo`060?ooo`00:P3oool200000:X0oooo00<0c<c<0?ooo`3oool00`3oool00`3<
c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<
c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool01@3<
c<`0oooo0?ooo`3oool0c<c<00X0oooo00<0c<c<0?ooo`3oool01P3oool002X0oooo00<000000?oo
o`3oool0Z@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?oo
o`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?oo
o`3oool00P3oool00`3<c<`0oooo0?ooo`060?ooo`050<c<c03oool0oooo0?ooo`3<c<`02`3oool0
0`3<c<`0oooo0?ooo`050?ooo`00:P3oool00`000000oooo0?ooo`2Y0?ooo`030<c<c03oool0oooo
00<0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo
00<0oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo
00H0oooo00D0c<c<0?ooo`3oool0oooo0<c<c00;0?ooo`030<c<c03oool0oooo00D0oooo000Z0?oo
o`030000003oool0oooo0:P0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`030?oo
o`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`060?oo
o`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool01@3<c<`0oooo0?ooo`3oool0
c<c<00/0oooo00<0c<c<0?ooo`3oool01@3oool002X0oooo00<000000?ooo`3oool0WP3oool20<c<
c0P0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo
00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo
0080oooo00<0c<c<0?ooo`3oool01P3oool01@3<c<`0oooo0?ooo`3oool0c<c<00/0oooo00<0c<c<
0?ooo`3oool01@3oool002X0oooo00<000000?ooo`3oool0WP3oool20<c<c0P0oooo00<0c<c<0?oo
o`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?oo
o`3oool00`3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?oo
o`3oool01P3oool01@3<c<`0oooo0?ooo`3oool0c<c<00/0oooo00<0c<c<0?ooo`3oool01@3oool0
02X0oooo00<000000?ooo`3oool0W@3oool00`3<c<`0oooo0<c<c0080?ooo`030<c<c03oool0oooo
00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo
00<0oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo
00L0oooo00@0c<c<0?ooo`3oool0c<c<2`3oool00`3<c<`0oooo0?ooo`050?ooo`00:P3oool00`00
0000oooo0?ooo`2M0?ooo`040<c<c03oool0oooo0<c<c0L0oooo00<0c<c<0?ooo`3oool01@3oool0
0`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3oool0
0`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool0
103<c<`0oooo0?ooo`3<c<`;0?ooo`030<c<c03oool0oooo00D0oooo000Z0?ooo`800000WP3oool0
103<c<`0oooo0?ooo`3<c<`70?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3o
ool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01P3o
ool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00L0oooo00@0c<c<0?ooo`3oool0c<c<
2`3oool00`3<c<`0oooo0?ooo`050?ooo`00:P3oool00`000000oooo0?ooo`2M0?ooo`040<c<c03o
ool0oooo0<c<c0L0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<
c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<
c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool0103<c<`0oooo0?ooo`3<c<`;0?ooo`03
0<c<c03oool0oooo00D0oooo000Z0?ooo`030000003oool0oooo09d0oooo00@0c<c<0?ooo`3oool0
c<c<1`3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3o
ool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?ooo`3o
ool00P3oool00`3<c<`0oooo0?ooo`070?ooo`040<c<c03oool0oooo0<c<c0/0oooo00<0c<c<0?oo
o`3oool01@3oool002X0oooo00<000000?ooo`3oool0T@3oool20<c<c0X0oooo00@0c<c<0?ooo`3o
ool0c<c<1`3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?oo
o`3oool01P3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00H0oooo00<0c<c<0?oo
o`3oool00P3oool00`3<c<`0oooo0?ooo`070?ooo`040<c<c03oool0oooo0<c<c0/0oooo00<0c<c<
0?ooo`3oool01@3oool002X0oooo00<000000?ooo`3oool0T@3oool20<c<c0T0oooo00D0c<c<0?oo
o`3oool0oooo0<c<c0070?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00`3oool0
0`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool0
0`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00L0oooo00@0c<c<0?ooo`3oool0c<c<2`3o
ool00`3<c<`0oooo0?ooo`050?ooo`001`3oool400000080oooo00D000000?ooo`3oool0oooo0000
00040?ooo`D000000P3oool2000000/0oooo00<000000?ooo`3oool0T03oool00`3<c<`0oooo0<c<
c0090?ooo`050<c<c03oool0oooo0?ooo`3<c<`01`3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<
c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01P3oool01@3<c<`0oooo0?ooo`3oool0c<c<00T0
oooo00D0c<c<0?ooo`3oool0oooo0<c<c00:0?ooo`040<c<c03oool0oooo0<c<c0/0oooo00<0c<c<
0?ooo`3oool01@3oool000L0oooo00<000000?ooo`3oool0103oool00`000000oooo000000070?oo
o`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`080?ooo`030000003oool0
oooo0900oooo00<0c<c<0?ooo`3<c<`02@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0
oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0
oooo00H0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0090?ooo`050<c<c03oool0oooo0?ooo`3<c<`0
2P3oool0103<c<`0oooo0?ooo`3<c<`;0?ooo`030<c<c03oool0oooo00D0oooo00080?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool01P3oool01@000000oooo0?ooo`3oool000000080
oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`2@0?ooo`040<c<c03oool0oooo0<c<
c0P0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo
00D0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`060?ooo`050<c<c03oool0oooo
0?ooo`3<c<`02@3oool01@3<c<`0oooo0?ooo`3oool0c<c<00X0oooo00@0c<c<0?ooo`3oool0c<c<
2`3oool00`3<c<`0oooo0?ooo`050?ooo`0000<0oooo0000000000000P0000040?ooo`030000003o
ool0oooo0080oooo00<000000?ooo`0000001`3oool01@000000oooo0?ooo`3oool000000080oooo
00<000000?ooo`3oool0203oool200000940oooo00@0c<c<0?ooo`3oool0c<c<203oool00`3<c<`0
oooo0?ooo`020?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0
oooo0?ooo`030?ooo`030<c<c03oool0oooo00H0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0090?oo
o`050<c<c03oool0oooo0?ooo`3<c<`02P3oool0103<c<`0oooo0?ooo`3<c<`;0?ooo`030<c<c03o
ool0oooo00D0oooo00070?ooo`040000003oool0oooo00000080oooo00D000000?ooo`3oool0oooo
000000060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo08@0oooo0P3<c<`90?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool00`3<
c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool01P3oool00`3<
c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00L0oooo00D0c<c<0?ooo`3oool0oooo0<c<c009
0?ooo`050<c<c03oool0oooo0?ooo`3<c<`02P3oool0103<c<`0oooo0?ooo`3<c<`;0?ooo`030<c<
c03oool0oooo00D0oooo00080?ooo`8000003@3oool2000000@0oooo0P0000080?ooo`@00000Q@3o
ool30<c<c0T0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03o
ool0oooo00<0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03o
ool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00T0oooo
00D0c<c<0?ooo`3oool0oooo0<c<c00:0?ooo`040<c<c03oool0oooo0<c<c0/0oooo00<0c<c<0?oo
o`3oool01@3oool002L0oooo00@000000?ooo`0000000000Q@3oool0103<c<`0oooo0?ooo`3<c<`8
0?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`04
0?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01P3oool00`3<c<`0oooo0?ooo`02
0?ooo`030<c<c03oool0oooo00L0oooo00D0c<c<0?ooo`3oool0oooo0<c<c00:0?ooo`040<c<c03o
ool0oooo0<c<c0X0oooo00@0c<c<0?ooo`3oool0c<c<2`3oool00`3<c<`0oooo0?ooo`050?ooo`00
8P3oool300000080oooo100000250?ooo`040<c<c03oool0oooo0<c<c0P0oooo00<0c<c<0?ooo`3o
ool00P3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3o
ool00P3oool00`3<c<`0oooo0?ooo`060?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3o
ool01`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00X0oooo00@0c<c<0?ooo`3oool0c<c<2P3oool0
103<c<`0oooo0?ooo`3<c<`;0?ooo`030<c<c03oool0oooo00D0oooo000W0?ooo`040000003oool0
oooo000008D0oooo00@0c<c<0?ooo`3oool0c<c<203oool00`3<c<`0oooo0?ooo`020?ooo`030<c<
c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`020?ooo`030<c<
c03oool0oooo00H0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`070?ooo`050<c<
c03oool0oooo0?ooo`3<c<`02P3oool0103<c<`0oooo0?ooo`3<c<`:0?ooo`040<c<c03oool0oooo
0<c<c0/0oooo00<0c<c<0?ooo`3oool01@3oool002P0oooo0`00001i0?ooo`80c<c<2@3oool01@3<
c<`0oooo0?ooo`3oool0c<c<00P0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`04
0?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`07
0?ooo`050<c<c03oool0oooo0?ooo`3<c<`02@3oool01@3<c<`0oooo0?ooo`3oool0c<c<00X0oooo
00@0c<c<0?ooo`3oool0c<c<2P3oool0103<c<`0oooo0?ooo`3<c<`;0?ooo`030<c<c03oool0oooo
00D0oooo000Z0?ooo`030000003oool0oooo07H0oooo0`3<c<`90?ooo`050<c<c03oool0oooo0?oo
o`3<c<`0203oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?oo
o`3oool0103oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00L0oooo00D0c<c<0?oo
o`3oool0oooo0<c<c0090?ooo`050<c<c03oool0oooo0?ooo`3<c<`02P3oool0103<c<`0oooo0?oo
o`3<c<`:0?ooo`040<c<c03oool0oooo0<c<c0`0oooo00<0c<c<0?ooo`3oool0103oool002X0oooo
0P00001g0?ooo`040<c<c03oool0oooo0<c<c0P0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0
oooo0?ooo`050?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0
oooo0?ooo`040?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool01@3<c<`0
oooo0?ooo`3oool0c<c<00T0oooo00D0c<c<0?ooo`3oool0oooo0<c<c00:0?ooo`040<c<c03oool0
oooo0<c<c0X0oooo00@0c<c<0?ooo`3oool0c<c<303oool00`3<c<`0oooo0?ooo`040?ooo`00:P3o
ool00`000000oooo0?ooo`1f0?ooo`040<c<c03oool0oooo0<c<c0P0oooo00<0c<c<0?ooo`3oool0
0P3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0
0`3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool0
1`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00T0oooo00D0c<c<0?ooo`3oool0oooo0<c<c00:0?oo
o`030<c<c03oool0c<c<00/0oooo00<0c<c<0?ooo`3<c<`03@3oool00`3<c<`0oooo0?ooo`040?oo
o`00:P3oool00`000000oooo0?ooo`1Z0?ooo`80c<c<2@3oool01@3<c<`0oooo0?ooo`3oool0c<c<
00P0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo
00@0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo
0080oooo00<0c<c<0?ooo`3oool01`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00T0oooo00D0c<c<
0?ooo`3oool0oooo0<c<c00:0?ooo`030<c<c03oool0c<c<00/0oooo00<0c<c<0?ooo`3<c<`03@3o
ool00`3<c<`0oooo0?ooo`040?ooo`00:P3oool00`000000oooo0?ooo`1Y0?ooo`<0c<c<2@3oool0
1@3<c<`0oooo0?ooo`3oool0c<c<00L0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0?oo
o`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?oo
o`050?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool01@3<c<`0oooo0?oo
o`3oool0c<c<00X0oooo00@0c<c<0?ooo`3oool0c<c<2P3oool00`3<c<`0oooo0<c<c00<0?ooo`80
c<c<3@3oool00`3<c<`0oooo0?ooo`040?ooo`00:P3oool00`000000oooo0?ooo`1Y0?ooo`040<c<
c03oool0c<c<0<c<c0P0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0070?ooo`030<c<c03oool0oooo
00<0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo
0080oooo00<0c<c<0?ooo`3oool01@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo
00L0oooo00D0c<c<0?ooo`3oool0oooo0<c<c00:0?ooo`030<c<c03oool0c<c<00/0oooo00<0c<c<
0?ooo`3<c<`0303oool20<c<c0d0oooo00<0c<c<0?ooo`3oool0103oool002X0oooo00<000000?oo
o`3oool0G@3oool00`3<c<`0oooo0?ooo`080?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool0
0`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool0
0`3<c<`0oooo0?ooo`030?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool00P3oool0
0`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01`3oool0
1@3<c<`0oooo0?ooo`3oool0c<c<00X0oooo00<0c<c<0?ooo`3<c<`02`3oool00`3<c<`0oooo0<c<
c00<0?ooo`80c<c<3@3oool00`3<c<`0oooo0?ooo`040?ooo`00:P3oool00`000000oooo0?ooo`1L
0?ooo`<0c<c<2@3oool01@3<c<`0oooo0?ooo`3oool0c<c<00P0oooo00<0c<c<0?ooo`3oool00P3o
ool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool00P3o
ool00`3<c<`0oooo0?ooo`050?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01@3o
ool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo00L0oooo00D0c<c<0?ooo`3oool0oooo
0<c<c00:0?ooo`030<c<c03oool0c<c<00/0oooo00<0c<c<0?ooo`3<c<`0303oool20<c<c0d0oooo
00<0c<c<0?ooo`3oool0103oool002X0oooo00<000000?ooo`3oool0G03oool0103<c<`0oooo0?oo
o`3<c<`80?ooo`050<c<c03oool0oooo0?ooo`3<c<`01`3oool00`3<c<`0oooo0?ooo`030?ooo`03
0<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`020?ooo`03
0<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?ooo`060?ooo`04
0<c<c03oool0oooo0<c<c0X0oooo00@0c<c<0?ooo`3oool0c<c<2`3oool00`3<c<`0oooo0<c<c00;
0?ooo`030<c<c03oool0c<c<00`0oooo0P3<c<`<0?ooo`030000003<c<`0000000D0oooo000Z0?oo
o`800000D@3oool20<c<c0T0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0080?ooo`030<c<c03oool0
oooo0080oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0
oooo00<0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0
oooo00H0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0080?ooo`040<c<c03oool0oooo0<c<c0X0oooo
00@0c<c<0?ooo`3oool0c<c<2`3oool00`3<c<`0oooo0<c<c00;0?ooo`030<c<c03oool0c<c<00`0
oooo0P3<c<`;0?ooo`80000000<0c<c<000000000000103oool002X0oooo00<000000?ooo`3oool0
C`3oool00`3<c<`0oooo0<c<c0090?ooo`050<c<c03oool0oooo0?ooo`3<c<`01`3oool00`3<c<`0
oooo0?ooo`030?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0
oooo0?ooo`030?ooo`030<c<c03oool0oooo00@0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0
oooo0?ooo`060?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool0103<c<`0oooo0?ooo`3<c<`:
0?ooo`040<c<c03oool0oooo0<c<c0/0oooo00<0c<c<0?ooo`3<c<`0303oool20<c<c0`0oooo0P3<
c<`;0?ooo`D00000103oool002X0oooo00<000000?ooo`3oool0C`3oool0103<c<`0oooo0?ooo`3<
c<`80?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?oo
o`040?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0oooo0?oo
o`020?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0oooo0?oo
o`060?ooo`050<c<c03oool0oooo0?ooo`3<c<`0203oool0103<c<`0oooo0?ooo`3<c<`;0?ooo`03
0<c<c03oool0c<c<00/0oooo00<0c<c<0?ooo`3<c<`0303oool20<c<c0/0oooo00<000000<c<c03<
c<`01`3oool9000000@0oooo000Z0?ooo`030000003oool0oooo04h0oooo00D0c<c<0?ooo`3oool0
oooo0<c<c0080?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0
oooo0?ooo`040?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3oool0103oool00`3<c<`0
oooo0?ooo`020?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?ooo`3oool00P3oool00`3<c<`0
oooo0?ooo`060?ooo`040<c<c03oool0oooo0<c<c0T0oooo00@0c<c<0?ooo`3oool0c<c<2`3oool0
0`3<c<`0oooo0<c<c00;0?ooo`030<c<c03oool0c<c<00`0oooo0P3<c<`:0?ooo`8000000P3<c<`0
0`000000oooo00000004000000D0oooo0`0000050?ooo`00:P3oool00`000000oooo0?ooo`0U0?oo
o`800000103oool2000000<0oooo1000000J0?ooo`030<c<c03oool0oooo0080oooo00@0c<c<0?oo
o`00000000000`3oool00`3<c<`00000000000030?ooo`030<c<c03oool00000008000000P3oool0
0`3<c<`0oooo0?ooo`040?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3oool01@3oool0
0`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0oooo0080oooo0P000000103oool0c<c<0?ooo`3o
ool2000000030<c<c03oool0oooo00<000000`3oool0103<c<`0oooo0?ooo`3<c<`90?ooo`040<c<
c03oool0oooo0<c<c0/0oooo00<0c<c<0?ooo`3<c<`00P0000040?ooo`8000000`3oool00`3<c<`0
00000<c<c00;0?ooo`030000003<c<`0c<c<00X0oooo1P0000090?ooo`D00000103oool002X0oooo
00<000000?ooo`3oool0903oool010000000oooo0?ooo`0000080?ooo`030000003oool0oooo01X0
oooo00<0c<c<0?ooo`3oool00`3oool01@3<c<`000000?ooo`3oool000000080oooo00<0c<c<0?oo
o`3oool0103oool00`3<c<`0oooo000000020?ooo`030<c<c03oool0oooo00D0oooo00<0c<c<0?oo
o`3oool00P3oool00`3<c<`0oooo0?ooo`060?ooo`050<c<c03oool0oooo0?ooo`3<c<`00`3oool0
1P000000oooo0?ooo`000000oooo0<c<c0<0oooo00@0c<c<0?ooo`3oool000000P3oool010000000
oooo0?ooo`3<c<`20?ooo`030<c<c03oool0oooo00L0oooo00@0c<c<0?ooo`3oool0c<c<2`3oool0
0`3<c<`0oooo0<c<c0020?ooo`030000003oool0oooo00H0oooo00@0c<c<0?ooo`3<c<`000002@3o
ool200000080c<c<00<000000?ooo`3oool00`3oool9000000`0oooo00<000000?ooo`3oool0103o
ool002X0oooo00<000000?ooo`3oool0903oool010000000oooo0?ooo`0000090?ooo`030000003o
ool0oooo01T0oooo00<0c<c<0?ooo`3oool00`3oool01@3<c<`000000?ooo`3oool000000080oooo
00<0c<c<0?ooo`3oool00`3oool00`000000c<c<00000002000000030?ooo`3<c<`0oooo00L0oooo
00D0c<c<0?ooo`3oool0oooo0<c<c0080?ooo`050<c<c03oool0oooo0?ooo`3<c<`00`3oool01P00
0000oooo0?ooo`000000oooo0<c<c0<0oooo00@0c<c<0?ooo`3oool000000P3oool010000000oooo
0?ooo`3<c<`20?ooo`030<c<c03oool0oooo00L0oooo00@0c<c<0?ooo`3oool0c<c<2`3oool00`3<
c<`0oooo0<c<c0020?ooo`030000003oool0oooo00H0oooo00@000000<c<c03<c<`000002@3oool2
00000080c<c<1P0000050?ooo`<000003@3oool00`000000oooo0?ooo`040?ooo`00:P3oool20000
02D0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`0H0?ooo`030<c<c03oool0
oooo00@0oooo00H0c<c<0?ooo`3oool000000?ooo`3<c<`60?ooo`040000003<c<`0oooo00000080
oooo00<0c<c<0?ooo`3oool01P3oool01@3<c<`0oooo0?ooo`3oool0c<c<00P0oooo00D0c<c<0?oo
o`3oool0oooo0<c<c0030?ooo`060000003oool0oooo0000003oool0c<c<0`3oool00`3<c<`0oooo
0?ooo`03000000<0oooo00@0c<c<0?ooo`3oool0c<c<2P3oool00`3<c<`0oooo0<c<c00;0?ooo`03
0<c<c03oool0c<c<0080oooo00<000000?ooo`3oool01@3oool2000000030<c<c0000000000000H0
oooo2000000J0?ooo`030000003oool0oooo00@0oooo000Z0?ooo`030000003oool0oooo02@0oooo
00@000000?ooo`3oool00000203oool010000000oooo0?ooo`00000H0?ooo`030<c<c03oool0oooo
00D0oooo00H0c<c<0?ooo`3oool000000?ooo`3<c<`70?ooo`030<c<c03oool000000080oooo00<0
c<c<0?ooo`3oool01P3oool01@3<c<`0oooo0?ooo`3oool0c<c<00P0oooo00@0c<c<0?ooo`3oool0
c<c<103oool01P000000oooo0?ooo`000000oooo0<c<c080oooo00<0c<c<0?ooo`3oool00P3oool0
0`000000oooo0?ooo`030?ooo`030<c<c03oool0c<c<00X0oooo00<0c<c<0?ooo`3<c<`02`3oool2
0<c<c0040000003oool0oooo000000L0oooo2`0000040?ooo`<000006`3oool00`000000oooo0?oo
o`040?ooo`00:P3oool00`000000oooo0?ooo`0U0?ooo`8000002P3oool2000001T0oooo00<0c<c<
0?ooo`3oool01P3oool0103<c<`000000?ooo`3<c<`90?ooo`040<c<c0000000oooo0<c<c0T0oooo
00@0c<c<0?ooo`3oool0c<c<2@3oool0103<c<`0oooo0?ooo`3<c<`50?ooo`8000000P3oool0103<
c<`0oooo0?ooo`3<c<`40?ooo`<000000`3oool00`3<c<`0oooo0<c<c00:0?ooo`80c<c<00<00000
0?ooo`3oool0203oool2000000030<c<c0000000000000h000009`3oool2000000H0oooo000Z0?oo
o`030000003oool0oooo04`0oooo00<0c<c<0?ooo`3oool01P3oool0103<c<`0oooo0?ooo`3<c<`9
0?ooo`040<c<c03oool0oooo0<c<c0T0oooo00@0c<c<0?ooo`3oool0c<c<2@3oool0103<c<`0oooo
0?ooo`3<c<`:0?ooo`030<c<c03oool0c<c<00X0oooo0P3<c<`00`000000oooo0?ooo`070?ooo`03
0000003<c<`0c<c<008000001`3oool7000000X0oooo0`00000`0?ooo`00:P3oool00`000000oooo
0?ooo`1;0?ooo`030<c<c03oool0oooo00L0oooo00@0c<c<0?ooo`3oool0c<c<2@3oool0103<c<`0
oooo0?ooo`3<c<`90?ooo`030<c<c03oool0c<c<00/0oooo00<0c<c<0000003<c<`02P3oool20<c<
c0030000003oool0oooo00L0oooo00<000000<c<c03<c<`00P0000060?ooo`h000000P3oool50000
03d0oooo000Z0?ooo`030000003oool0oooo0400oooo0`0000020?ooo`80c<c<0P3oool200000003
0<c<c03oool0oooo00L0oooo0P3<c<`00`000000c<c<0?ooo`090?ooo`030<c<c0000000c<c<00X0
oooo0P3<c<`00`000000oooo0?ooo`070?ooo`030000003<c<`0c<c<00800000203oool00`000000
c<c<0<c<c002000000030?ooo`0000000000010000000P3oool5000000X0oooo0`00000n0?ooo`00
:@3oool?000000030?ooo`000000000000@0000000D0oooo00000000000000000?ooo`0300000005
0?ooo`3<c<`0c<c<0<c<c03oool00`0000001@3oool0c<c<0<c<c03<c<`0oooo00<0000000<0oooo
0<c<c03<c<`00P3<c<`0103oool000000000000000040<c<c0040?ooo`0000000000000000D0c<c<
100000000`3<c<`0oooo0<c<c0020<c<c080000000<0c<c<0000003oool01`3oool00`000000c<c<
0<c<c002000000P0oooo00<000000<c<c03<c<`00P0000080?ooo`030000003<c<`0c<c<01d00000
1`3oool5000000T0oooo0`00001<0?ooo`00:03ooolO00000080c<c<00D0oooo0<c<c03<c<`0c<c<
000000020<c<c0<0oooo0P3<c<`00`000000c<c<0<c<c0040?ooo`D0c<c<0`3oool40<c<c3T00000
203oool5000000P0oooo1@0000090?ooo`<00000F@3oool002@0ooooo000000002P0oooo0P000000
0`3<c<`000000000000K000000<0oooo1@0000030?ooo`D0000000@0oooo0000003oool0oooo1@00
00030?ooo`D00000103oool3000000030?ooo`000000oooo00<0oooo0`0000040?ooo`030000003o
ool0oooo00<0oooo0`0000030?ooo`030000003oool0oooo00@0oooo0`0000020?ooo`030000003o
ool0oooo00D0oooo0`0000000`3oool000000?ooo`0:0?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00@0oooo000X
0?oooa00000000<0oooo000000000000100000001@3oool00000000000000000oooo00<000001@3o
ool3000000D0oooo0`0000020?ooo`040000003oool0oooo0?ooo`<000001@3oool300000240oooo
00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo02d0oooo
00<000000?ooo`3oool0103oool002T0oooo1@00003b0?ooo`00:P3oool300000?<0oooo000Z0?oo
o`030000003oool0oooo0?<0oooo000Z0?ooo`030000003oool0oooo0?<0oooo003o0?ooob40oooo
003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo
003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.176385, -1.03253*^-05, \
0.00418603, 6.43332*^-08}}]
}, Open  ]],

Cell["\<\
From the plot, it is quite apparent that when the time steps get large, the \
default solution output has much larger error between time steps.  The dense \
output solution represents the solution computed by the solver even between \
time steps.  Keep in mind, however, that the dense output solution takes up \
much more space:\
\>", "Text"],

Cell["\<\
This compares the sizes of the default and dense output solutions.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ByteCount\  /@ \ {f1[t], \ f1\_dense[t]}\)], "Input",
  CellLabel->"In[43]:="],

Cell[BoxData[
    \({3104, 15952}\)], "Output",
  CellLabel->"Out[43]="]
}, Open  ]],

Cell["\<\
When the quantity you want to derive from the solution is complicated, you \
can ensure that it is locally kept within tolerances by giving it as an \
algebraic quantity, forcing the solver to keep its error in control.\
\>", "Text"],

Cell["\<\
This adds a dependent variable with an algebraic equation that sets the \
dependent variable equal to the quantity of interest.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(f2[t_]\  = \ 
      First[g[t]\  /. \ 
          NDSolve[{eqns, \ ics, \ g[t]\  \[Equal] \ f[t]}, \ {x, y, g}, {t, 
              0, 1}]]\)], "Input",
  CellLabel->"In[44]:="],

Cell[BoxData[
    RowBox[{
      TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
        False,
        Editable->False], "[", "t", "]"}]], "Output",
  CellLabel->"Out[44]="]
}, Open  ]],

Cell["This computes the same solution with dense output", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(f2\_dense[t_]\  = \ 
      First[g[t]\  /. \ 
          NDSolve[{eqns, \ ics, \ g[t]\  \[Equal] \ f[t]}, \ {x, y, g}, {t, 
              0, 1}, \ Method \[Rule] {"\<IDA\>", \ "\<DenseOutput\>" \[Rule] 
                  True}]]\)], "Input",
  CellLabel->"In[45]:="],

Cell[BoxData[
    RowBox[{
      TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
        False,
        Editable->False], "[", "t", "]"}]], "Output",
  CellLabel->"Out[45]="]
}, Open  ]],

Cell["\<\
This makes a plot comparing the error for all four solutions.  The time steps \
for IDA are shown as blue points and the dense output from IDA is shown in \
blue with the default output shown in light blue.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(t2\  = \(InterpolatingFunctionCoordinates[
            Head[f2[t]]]\)[\([1]\)];\)\), "\[IndentingNewLine]", 
    \(\(Show[
        Block[{$DisplayFunction\  = \ Identity}, {pode, \ 
            ListPlot[Transpose[{t2, \ f\_exact[t2]\  - \ f2[t2]}], \ 
              PlotStyle \[Rule] {RGBColor[0, 0, 1], PointSize[0.02]}], 
            Plot[f\_exact[t] - \ f2[t], {t, 0, 1}, \ 
              PlotStyle \[Rule] RGBColor[ .7,  .7, 1]], 
            Plot[f\_exact[t] - \ f2\_dense[t], {t, 0, 1}, \ 
              PlotStyle \[Rule] RGBColor[0, 0, 1]]}], \ 
        PlotRange \[Rule] {{0, 1}, \ 1*^-7\ {\(-1\), 1}}];\)\)}], "Input",
  CellLabel->"In[46]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0 1 0.309017 3.09017e+006 [
[.2 .29652 -9 -9 ]
[.2 .29652 9 0 ]
[.4 .29652 -9 -9 ]
[.4 .29652 9 0 ]
[.6 .29652 -9 -9 ]
[.6 .29652 9 0 ]
[.8 .29652 -9 -9 ]
[.8 .29652 9 0 ]
[1 .29652 -3 -9 ]
[1 .29652 3 0 ]
[-0.0125 0 -39 -6.25 ]
[-0.0125 0 0 6.25 ]
[-0.0125 .07725 -51 -6.25 ]
[-0.0125 .07725 0 6.25 ]
[-0.0125 .15451 -39 -6.25 ]
[-0.0125 .15451 0 6.25 ]
[-0.0125 .23176 -51 -6.25 ]
[-0.0125 .23176 0 6.25 ]
[-0.0125 .38627 -45 -6.25 ]
[-0.0125 .38627 0 6.25 ]
[-0.0125 .46353 -33 -6.25 ]
[-0.0125 .46353 0 6.25 ]
[-0.0125 .54078 -45 -6.25 ]
[-0.0125 .54078 0 6.25 ]
[-0.0125 .61803 -33 -6.25 ]
[-0.0125 .61803 0 6.25 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.2 .30902 m
.2 .31527 L
s
[(0.2)] .2 .29652 0 1 Mshowa
.4 .30902 m
.4 .31527 L
s
[(0.4)] .4 .29652 0 1 Mshowa
.6 .30902 m
.6 .31527 L
s
[(0.6)] .6 .29652 0 1 Mshowa
.8 .30902 m
.8 .31527 L
s
[(0.8)] .8 .29652 0 1 Mshowa
1 .30902 m
1 .31527 L
s
[(1)] 1 .29652 0 1 Mshowa
.125 Mabswid
.05 .30902 m
.05 .31277 L
s
.1 .30902 m
.1 .31277 L
s
.15 .30902 m
.15 .31277 L
s
.25 .30902 m
.25 .31277 L
s
.3 .30902 m
.3 .31277 L
s
.35 .30902 m
.35 .31277 L
s
.45 .30902 m
.45 .31277 L
s
.5 .30902 m
.5 .31277 L
s
.55 .30902 m
.55 .31277 L
s
.65 .30902 m
.65 .31277 L
s
.7 .30902 m
.7 .31277 L
s
.75 .30902 m
.75 .31277 L
s
.85 .30902 m
.85 .31277 L
s
.9 .30902 m
.9 .31277 L
s
.95 .30902 m
.95 .31277 L
s
.25 Mabswid
0 .30902 m
1 .30902 L
s
0 0 m
.00625 0 L
s
gsave
-0.0125 0 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
0 .07725 m
.00625 .07725 L
s
gsave
-0.0125 .07725 -51 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-7.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-8) show
grestore
0 .15451 m
.00625 .15451 L
s
gsave
-0.0125 .15451 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-8) show
grestore
0 .23176 m
.00625 .23176 L
s
gsave
-0.0125 .23176 -51 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-2.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-8) show
grestore
0 .38627 m
.00625 .38627 L
s
gsave
-0.0125 .38627 -45 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-8) show
grestore
0 .46353 m
.00625 .46353 L
s
gsave
-0.0125 .46353 -33 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-8) show
grestore
0 .54078 m
.00625 .54078 L
s
gsave
-0.0125 .54078 -45 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(7.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-8) show
grestore
0 .61803 m
.00625 .61803 L
s
gsave
-0.0125 .61803 -33 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.125 Mabswid
0 .01545 m
.00375 .01545 L
s
0 .0309 m
.00375 .0309 L
s
0 .04635 m
.00375 .04635 L
s
0 .0618 m
.00375 .0618 L
s
0 .09271 m
.00375 .09271 L
s
0 .10816 m
.00375 .10816 L
s
0 .12361 m
.00375 .12361 L
s
0 .13906 m
.00375 .13906 L
s
0 .16996 m
.00375 .16996 L
s
0 .18541 m
.00375 .18541 L
s
0 .20086 m
.00375 .20086 L
s
0 .21631 m
.00375 .21631 L
s
0 .24721 m
.00375 .24721 L
s
0 .26266 m
.00375 .26266 L
s
0 .27812 m
.00375 .27812 L
s
0 .29357 m
.00375 .29357 L
s
0 .32447 m
.00375 .32447 L
s
0 .33992 m
.00375 .33992 L
s
0 .35537 m
.00375 .35537 L
s
0 .37082 m
.00375 .37082 L
s
0 .40172 m
.00375 .40172 L
s
0 .41717 m
.00375 .41717 L
s
0 .43262 m
.00375 .43262 L
s
0 .44807 m
.00375 .44807 L
s
0 .47898 m
.00375 .47898 L
s
0 .49443 m
.00375 .49443 L
s
0 .50988 m
.00375 .50988 L
s
0 .52533 m
.00375 .52533 L
s
0 .55623 m
.00375 .55623 L
s
0 .57168 m
.00375 .57168 L
s
0 .58713 m
.00375 .58713 L
s
0 .60258 m
.00375 .60258 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.02 w
0 .30902 Mdot
7e-005 .37415 Mdot
.00015 .43929 Mdot
.00251 .51436 Mdot
.00487 .58978 Mdot
.01477 .29242 Mdot
.02231 .30058 Mdot
.02986 .30904 Mdot
.0374 .31766 Mdot
.05248 .31617 Mdot
.06757 .28944 Mdot
.08265 .28687 Mdot
.09773 .28418 Mdot
.11282 .28142 Mdot
.1477 .24086 Mdot
.18259 .24646 Mdot
.21747 .23099 Mdot
.25236 .20742 Mdot
.28724 .18432 Mdot
.32213 .15807 Mdot
.37663 .01731 Mdot
.43114 .03473 Mdot
.48564 .00623 Mdot
.8 .8 .8 r
.5 Mabswid
0 .30902 m
.00123 .47246 L
.00236 .51348 L
.00488 .58979 L
s
.00488 .58979 m
.00577 .61803 L
s
.00885 .61803 m
.00942 .5891 L
.01067 .50315 L
.01201 .40545 L
.01328 .33038 L
.01444 .29456 L
.01573 .29267 L
.01696 .29371 L
.0183 .29556 L
.01975 .29797 L
.02043 .29901 L
.02116 .29994 L
.02249 .30059 L
.02378 .30118 L
.02497 .30252 L
.03054 .30918 L
.03176 .31009 L
.03306 .31185 L
.03542 .31585 L
.0367 .31739 L
.03787 .31753 L
.03926 .31598 L
.04057 .31366 L
.04179 .31142 L
.0431 .30945 L
.04386 .30868 L
.04455 .30826 L
.0459 .30827 L
.04722 .30932 L
.04797 .31028 L
.04867 .31136 L
.05016 .31386 L
.0509 .31496 L
.05157 .31573 L
.05281 .31607 L
.05415 .31391 L
.05529 .31055 L
.0565 .3062 L
.05791 .3009 L
.05925 .29634 L
.06045 .29302 L
.06176 .29046 L
.06244 .28958 L
.06318 .28899 L
.06449 .28867 L
.06529 .28883 L
.06603 .28909 L
.06671 .28932 L
.06743 .28944 L
.06876 .28858 L
Mistroke
.06998 .28659 L
.07142 .28372 L
.07273 .28135 L
.07347 .2803 L
.07417 .27958 L
.07491 .27913 L
.07572 .27904 L
.07642 .27929 L
.07716 .27988 L
.07849 .2816 L
.07994 .28399 L
.0807 .28521 L
.0815 .28624 L
.08239 .28685 L
.08321 .28664 L
.08397 .28581 L
.08481 .28443 L
.08748 .27915 L
.08821 .27797 L
.08899 .27702 L
.09038 .27621 L
.09167 .27657 L
.09288 .2778 L
.0942 .27983 L
.09563 .28224 L
.09689 .28383 L
.09826 .28397 L
.09956 .2823 L
.10076 .27994 L
.10185 .27765 L
.10301 .27553 L
.10428 .27391 L
.10544 .27329 L
.10669 .27361 L
.10805 .27503 L
.10933 .27705 L
.11051 .27911 L
.11124 .28023 L
.11193 .28102 L
.11322 .28082 L
.1146 .27075 L
.11539 .26039 L
.11611 .24838 L
.12158 .11843 L
.12459 .04769 L
.12607 .01978 L
Mfstroke
.12607 .01978 m
.12743 0 L
s
.13438 0 m
.13509 .00916 L
.13761 .05456 L
.13999 .10868 L
.14254 .1692 L
.14377 .19565 L
.14494 .21675 L
.14599 .23097 L
.14713 .2397 L
.1484 .2391 L
.14974 .22697 L
.15089 .20925 L
.15215 .18385 L
.15767 .04711 L
s
.15767 .04711 m
.15992 0 L
s
.17062 0 m
.17131 .01319 L
.17366 .06891 L
.17622 .13694 L
.17901 .20467 L
.17974 .21873 L
.18051 .2311 L
.18125 .23979 L
.18191 .24473 L
.18259 .24646 L
.18333 .24425 L
.18466 .23078 L
.18597 .20805 L
.18735 .17669 L
.18982 .11007 L
s
.18982 .11007 m
.19399 0 L
s
.20678 0 m
.2092 .06237 L
.21178 .13424 L
.2142 .19356 L
.21554 .21697 L
.2163 .22567 L
.217 .23015 L
.21823 .22841 L
.21953 .21404 L
.22067 .19308 L
.2219 .16409 L
.22411 .10119 L
.22671 .0219 L
s
.22671 .0219 m
.22751 0 L
s
.24298 0 m
.24555 .07017 L
.24687 .10961 L
.2483 .14853 L
.24963 .17843 L
.25088 .19843 L
.25155 .2047 L
.25227 .20741 L
.25355 .20089 L
.25428 .19129 L
.25507 .17713 L
.25647 .14353 L
s
.25647 .14353 m
.26101 0 L
s
.27935 0 m
.2811 .05688 L
.28343 .12675 L
.28476 .15794 L
.28543 .16979 L
.28616 .17905 L
.2869 .18385 L
.28769 .18317 L
.28844 .17711 L
.28911 .16759 L
.29043 .13989 L
.29186 .09944 L
.29444 .01048 L
s
.29444 .01048 m
.29474 0 L
s
.31545 0 m
.31632 .0308 L
.31782 .08069 L
.31915 .11788 L
.31988 .13416 L
.32056 .14608 L
.32134 .15502 L
.32219 .15798 L
.32292 .14928 L
.32371 .12503 L
.32447 .0875 L
.3252 .04027 L
s
.3252 .04027 m
.32568 0 L
s
.37573 0 m
.37628 .01556 L
.37763 .00132 L
s
.37763 .00132 m
.37766 0 L
s
.42984 0 m
.43085 .03338 L
.43211 .01675 L
s
.43211 .01675 m
.43241 0 L
s
.48521 0 m
.48545 .0056 L
s
.48545 .0056 m
.48604 0 L
s
0 g
0 .30909 m
.00488 .44216 L
.006 .6042 L
s
.006 .6042 m
.00627 .61803 L
s
.00885 .61803 m
.00942 .5891 L
.01069 .50158 L
.01204 .40378 L
.01317 .33572 L
.01441 .29486 L
.01574 .29267 L
.01649 .29321 L
.01717 .29395 L
.01849 .29587 L
.01975 .29797 L
.02106 .29983 L
.02225 .30058 L
.02353 .30099 L
.02491 .30243 L
.0262 .30443 L
.02738 .30647 L
.02851 .30814 L
.0297 .30903 L
.03262 .31257 L
.03332 .31301 L
.03406 .31387 L
.03539 .31585 L
.03656 .31728 L
.03787 .3177 L
.04057 .31884 L
.04187 .31949 L
.04261 .31978 L
.04328 .31996 L
.04452 .32007 L
.04583 .31982 L
.04651 .31955 L
.04723 .31917 L
.04853 .31831 L
.05147 .31638 L
.05279 .31615 L
.05353 .31592 L
.05422 .31549 L
.05548 .31417 L
.05681 .31207 L
.05912 .30703 L
.06166 .30029 L
.06307 .29653 L
.06462 .29293 L
.06532 .2916 L
.06607 .29047 L
.06742 .28946 L
.0683 .28939 L
.06912 .28932 L
.07066 .28914 L
Mistroke
.07208 .28893 L
.07362 .28864 L
.08481 .28667 L
.10579 .28286 L
.11088 .28167 L
.11223 .28147 L
.11294 .28141 L
.11369 .2813 L
.11507 .28093 L
.11634 .28037 L
.11783 .27947 L
.11948 .27818 L
.12233 .27521 L
.12792 .26725 L
.13784 .25034 L
.14025 .24678 L
.14282 .24369 L
.14431 .24235 L
.14565 .24148 L
.147 .24097 L
.14827 .24083 L
.14893 .24081 L
.14966 .2408 L
.15042 .24081 L
.15113 .24084 L
.1524 .24093 L
.15376 .24108 L
.15536 .24133 L
.15687 .24162 L
.15971 .24231 L
.16469 .24377 L
.16733 .24458 L
.17019 .24539 L
.17291 .24603 L
.1741 .24625 L
.1754 .24644 L
.17664 .24658 L
.17779 .24666 L
.17901 .24669 L
.1803 .24666 L
.18103 .24662 L
.18169 .24656 L
.18243 .24648 L
.1832 .24638 L
.18594 .24591 L
.18838 .24536 L
.19106 .24459 L
.19588 .24271 L
.20105 .24004 L
.2104 .23425 L
Mistroke
.2537 .20705 L
.27534 .19284 L
.28077 .18815 L
.28587 .18485 L
.29085 .18246 L
.29303 .18158 L
.29542 .18035 L
.29996 .17726 L
.30475 .17312 L
.31484 .16318 L
.32041 .15888 L
.32314 .15763 L
.32435 .15695 L
.32567 .15599 L
.32797 .15382 L
.3304 .15077 L
.33554 .14178 L
.34063 .12949 L
.34612 .11283 L
.35604 .07644 L
.36654 .03804 L
.36941 .02963 L
.37099 .02571 L
.37248 .02257 L
.37387 .02019 L
.3746 .01917 L
.37538 .01829 L
.37675 .01724 L
.37801 .01659 L
.38051 .0155 L
.38189 .01503 L
.38319 .01467 L
.3844 .01441 L
.38573 .01421 L
.38693 .01412 L
.38804 .01409 L
.38873 .01411 L
.38948 .01417 L
.39084 .01433 L
.39217 .01459 L
.39338 .01491 L
.39615 .01593 L
.39907 .01741 L
.40454 .02114 L
.40964 .02541 L
.4152 .03024 L
.41812 .03253 L
.41972 .03363 L
.42123 .03453 L
.42265 .03522 L
Mistroke
.42395 .0357 L
.42469 .0359 L
.4254 .03604 L
.42614 .03613 L
.42695 .03614 L
.42828 .03598 L
.42902 .03578 L
.42969 .03553 L
.431 .03481 L
.43224 .03399 L
.44265 .02808 L
.45202 .02414 L
.46216 .02051 L
.47213 .0162 L
.47779 .0128 L
.48308 .00864 L
Mfstroke
.48308 .00864 m
.49118 0 L
s
0 0 1 r
.02 w
0 .30896 Mdot
0 .30902 Mdot
0 .30902 Mdot
0 .30902 Mdot
0 .30902 Mdot
0 .30902 Mdot
0 .30902 Mdot
0 .30903 Mdot
0 .30906 Mdot
1e-005 .30918 Mdot
1e-005 .30966 Mdot
3e-005 .31159 Mdot
5e-005 .31932 Mdot
.0001 .35023 Mdot
.00015 .38114 Mdot
.0002 .41206 Mdot
.0003 .43352 Mdot
.0004 .43506 Mdot
.0005 .43285 Mdot
.0007 .43278 Mdot
.0009 .43289 Mdot
.0013 .4339 Mdot
.0021 .4418 Mdot
.0029 .45226 Mdot
.0037 .46358 Mdot
.0045 .47521 Mdot
.0061 .48715 Mdot
.0077 .48347 Mdot
.0093 .47667 Mdot
.0125 .46212 Mdot
.0157 .4559 Mdot
.0189 .45565 Mdot
.0221 .45776 Mdot
.0253 .46027 Mdot
.0317 .462 Mdot
.0381 .4605 Mdot
.0445 .45954 Mdot
.0509 .45947 Mdot
.0573 .46027 Mdot
.0701 .46242 Mdot
.08162 .46285 Mdot
.09314 .46944 Mdot
.10466 .46984 Mdot
.11618 .46194 Mdot
.1277 .45855 Mdot
.13922 .45577 Mdot
.15074 .44301 Mdot
.16226 .43673 Mdot
.17378 .45476 Mdot
.1853 .44797 Mdot
.19682 .42657 Mdot
.20834 .45732 Mdot
.21986 .46864 Mdot
.23138 .40723 Mdot
.2429 .44372 Mdot
.25442 .51872 Mdot
.26479 .43934 Mdot
.27516 .4391 Mdot
.28552 .47493 Mdot
.29486 .42621 Mdot
.30419 .40759 Mdot
.31352 .47289 Mdot
.32285 .44001 Mdot
.33218 .4225 Mdot
.34058 .45324 Mdot
.34898 .45381 Mdot
.35737 .45438 Mdot
.36577 .45491 Mdot
.37417 .45545 Mdot
.38257 .456 Mdot
.39097 .45658 Mdot
.40776 .4574 Mdot
.42456 .45754 Mdot
.44136 .45695 Mdot
.45815 .45634 Mdot
.47495 .45575 Mdot
.49174 .45539 Mdot
.50854 .45509 Mdot
.52534 .4548 Mdot
.54213 .45464 Mdot
.55893 .4547 Mdot
.57572 .45493 Mdot
.59252 .45529 Mdot
.60932 .45582 Mdot
.62611 .45658 Mdot
.64291 .45758 Mdot
.65971 .45884 Mdot
.6765 .46035 Mdot
.6933 .46214 Mdot
.71009 .46425 Mdot
.72689 .46668 Mdot
.74369 .46946 Mdot
.76048 .4726 Mdot
.77728 .47612 Mdot
.79407 .48004 Mdot
.81087 .48438 Mdot
.82767 .48915 Mdot
.84446 .49437 Mdot
.86126 .50006 Mdot
.87806 .50622 Mdot
.89485 .51287 Mdot
.91165 .52003 Mdot
.92844 .52769 Mdot
.94524 .53587 Mdot
.96204 .54459 Mdot
.97883 .55383 Mdot
.99563 .56361 Mdot
1 .5521 Mdot
.7 .7 1 r
.5 Mabswid
0 .30902 m
.00123 .43352 L
.00236 .44598 L
.00357 .46066 L
.00488 .48305 L
.00609 .48714 L
.00722 .48625 L
.00829 .47988 L
.00942 .47634 L
.01064 .46898 L
.01179 .46203 L
.01312 .46276 L
.01434 .45795 L
.01567 .45579 L
.01643 .45715 L
.01713 .45624 L
.0184 .45453 L
.01975 .45725 L
.02043 .45686 L
.02115 .45627 L
.02245 .45885 L
.02317 .45946 L
.02384 .45899 L
.02458 .45873 L
.02537 .46056 L
.02605 .46303 L
.02679 .46495 L
.02814 .46609 L
.02945 .46471 L
.03067 .46264 L
.03184 .46208 L
.03295 .46293 L
.03416 .46346 L
.03547 .46281 L
.03676 .46128 L
.03813 .46052 L
.03929 .46134 L
.04057 .46193 L
.04585 .46091 L
.04654 .46152 L
.04728 .4618 L
.04863 .46116 L
.04936 .4604 L
.05003 .45974 L
.05082 .45945 L
.05157 .45995 L
.05282 .4612 L
.05416 .46172 L
.05492 .46143 L
.05562 .46091 L
Mistroke
.05697 .46016 L
.05821 .46278 L
.05954 .47076 L
.06079 .47903 L
.06196 .4848 L
.06266 .48671 L
.06341 .48722 L
.06424 .48587 L
.06499 .48303 L
.06578 .47874 L
.06664 .47312 L
.06738 .46812 L
.0682 .46338 L
.06894 .46074 L
.06964 .46069 L
.07038 .46408 L
.07118 .46954 L
.07187 .47458 L
.0726 .47959 L
.07393 .4861 L
.07525 .48785 L
.07601 .4865 L
.0767 .4839 L
.07742 .48005 L
.0782 .47501 L
.07887 .47042 L
.07961 .46582 L
.08092 .46147 L
.08164 .46295 L
.08232 .46666 L
.08352 .47486 L
.08481 .48279 L
.08618 .48753 L
.08745 .48731 L
.08881 .48243 L
.08959 .4782 L
.09032 .47395 L
.09163 .46791 L
.09284 .46798 L
.09416 .47675 L
.09555 .48765 L
.09672 .49464 L
.09797 .49793 L
.09916 .4964 L
.10023 .49143 L
.10152 .48256 L
.10272 .47404 L
.10397 .46902 L
.10535 .47318 L
.10672 .48175 L
Mistroke
.10747 .48617 L
.10818 .48944 L
.10954 .49194 L
.11079 .48927 L
.1115 .48577 L
.11225 .48082 L
.11361 .47052 L
.11444 .4649 L
.11523 .46153 L
.11593 .46121 L
.11669 .46435 L
.11809 .47331 L
.1194 .48095 L
.12077 .48485 L
.12149 .48466 L
.12227 .48267 L
.12365 .47527 L
.12495 .46593 L
.12612 .45884 L
.1274 .45731 L
.12868 .46514 L
.12985 .47435 L
.13114 .48261 L
.13249 .48642 L
.13368 .4847 L
.13478 .47915 L
.13599 .47015 L
.13728 .46023 L
.13801 .45621 L
.13869 .45475 L
.13943 .45674 L
.14022 .46176 L
.14165 .47325 L
.14299 .48207 L
.14418 .48558 L
.14546 .48352 L
.14619 .47968 L
.14688 .47453 L
.14818 .46226 L
.14954 .44936 L
.15083 .44288 L
.15154 .44346 L
.15231 .44648 L
.15366 .45416 L
.15495 .46055 L
.1563 .46313 L
.15746 .46098 L
.15874 .4545 L
.16021 .44431 L
.16094 .43987 L
Mistroke
.16161 .43719 L
.16238 .43695 L
.16322 .44067 L
.16394 .44591 L
.16469 .4524 L
.16596 .46295 L
.16733 .47062 L
.16811 .47244 L
.16883 .47233 L
.16955 .47064 L
.17024 .46782 L
.17154 .46071 L
.17223 .45717 L
.17296 .45466 L
.17429 .45692 L
.17551 .46663 L
.17662 .47717 L
.1778 .48678 L
.17909 .49236 L
.1803 .49143 L
.18168 .48319 L
.18246 .47586 L
.18318 .46811 L
.18448 .45436 L
.18522 .44847 L
.18591 .44581 L
.1871 .44721 L
.18839 .45231 L
.18976 .4568 L
.19103 .45714 L
.19215 .45354 L
.19335 .44611 L
.19439 .43808 L
.19552 .43013 L
.19684 .4266 L
.19809 .43257 L
.19931 .4428 L
.20043 .45229 L
.20161 .45972 L
.20272 .46303 L
.20393 .46246 L
.20524 .45832 L
.2066 .45373 L
.2079 .4546 L
.20906 .46459 L
.21033 .4817 L
.21164 .49952 L
.21302 .51314 L
.21374 .51692 L
.21453 .51802 L
Mistroke
.21525 .51623 L
.21592 .51228 L
.21725 .49924 L
.21846 .48408 L
.21913 .47594 L
.21986 .46862 L
.22118 .46394 L
.22236 .46611 L
.22366 .46955 L
.22502 .46971 L
.22629 .46422 L
.22748 .4538 L
.22857 .44069 L
.22985 .4235 L
.23102 .41016 L
.2323 .40387 L
.23368 .40643 L
.23495 .41175 L
.23615 .41613 L
.23683 .41773 L
.23755 .41864 L
.23884 .41878 L
.23956 .41875 L
.24023 .41934 L
.24097 .42154 L
.24176 .42694 L
.2446 .48345 L
.24615 .51914 L
.24692 .53339 L
.24762 .54375 L
.24886 .55477 L
.2502 .55588 L
.25159 .547 L
.25228 .53996 L
.25305 .53152 L
.25377 .52404 L
.25445 .51856 L
.25572 .51573 L
.25708 .51844 L
.25786 .5198 L
.25857 .51983 L
.25976 .51578 L
.26106 .50406 L
.26243 .48393 L
.26371 .46009 L
.26442 .44617 L
.26521 .43234 L
.26594 .42272 L
.26663 .41614 L
.26792 .40759 L
Mistroke
.2693 .40149 L
.27008 .39895 L
.27092 .39718 L
.27172 .39699 L
.27245 .39882 L
.27312 .40288 L
.27385 .41094 L
.27461 .42474 L
.27531 .44386 L
.27663 .4816 L
.27732 .49904 L
.27805 .51463 L
.2794 .53346 L
.28065 .5384 L
.28184 .53222 L
.28296 .51842 L
.28418 .49795 L
.28551 .47518 L
.28684 .46232 L
.28757 .45971 L
.28825 .45867 L
.28954 .45755 L
.29076 .45477 L
.29145 .45179 L
.2922 .44732 L
.2935 .43714 L
.29429 .43054 L
.29503 .42502 L
.2957 .42151 L
.29643 .41874 L
.29891 .40979 L
.3002 .40348 L
.30157 .39746 L
.30288 .39674 L
.30406 .40583 L
.30525 .42425 L
.30637 .44129 L
.30771 .45737 L
.30838 .463 L
.30911 .46716 L
.31042 .46997 L
.31163 .46951 L
.31237 .46941 L
.31306 .47073 L
.31437 .48009 L
.31577 .49519 L
.31651 .50223 L
.31731 .5075 L
.31869 .50837 L
.31998 .49795 L
Mistroke
.32074 .48672 L
.32146 .47292 L
.32281 .44093 L
.32407 .41577 L
.32524 .40245 L
.32653 .39452 L
.3279 .39087 L
.32868 .39071 L
.32939 .39204 L
.33018 .3957 L
.331 .40306 L
.33173 .41347 L
.3324 .42735 L
.33319 .4438 L
.33393 .45731 L
.33531 .47469 L
.33657 .48049 L
.33793 .47655 L
.33869 .47076 L
.3394 .46412 L
.34064 .45278 L
.34194 .45026 L
.34305 .45492 L
.34426 .46165 L
.34557 .46637 L
.34631 .46676 L
.34697 .46549 L
.34827 .45895 L
.3495 .45056 L
.35065 .4475 L
.35187 .44715 L
.3529 .44718 L
.35405 .44677 L
.35528 .44647 L
.3566 .4492 L
.35784 .45881 L
.35898 .46929 L
.36032 .47769 L
.36106 .47952 L
.36173 .47921 L
.36305 .47368 L
.36429 .46463 L
.36499 .45935 L
.36574 .45507 L
.36707 .45406 L
.36781 .45592 L
.3685 .45802 L
.36926 .46002 L
.37007 .46122 L
.37143 .46044 L
Mistroke
.3722 .45875 L
.3729 .45695 L
.37421 .45547 L
.37559 .45969 L
.37674 .46445 L
.37797 .46739 L
.37933 .4663 L
.38061 .46183 L
.38193 .45689 L
.38313 .45672 L
.38452 .46176 L
.38582 .46626 L
.38703 .46738 L
.38815 .46528 L
.38938 .46071 L
.39067 .45678 L
.39198 .46587 L
.39271 .48168 L
.39338 .50045 L
.39591 .58179 L
.3966 .60074 L
.39733 .61736 L
Mfstroke
.39733 .61736 m
.39737 .61803 L
s
.40135 .61803 m
.40167 .61286 L
.40296 .57906 L
.40438 .53354 L
.40575 .49045 L
.40653 .47157 L
.40723 .4605 L
.40859 .46322 L
.40987 .49147 L
.41105 .52847 L
.41233 .57067 L
.41371 .60936 L
s
.41371 .60936 m
.41414 .61803 L
s
.41831 .61803 m
.41968 .58319 L
.42095 .54239 L
.42235 .49712 L
.42312 .4767 L
.42383 .46323 L
.42516 .46039 L
.42586 .47133 L
.42661 .48971 L
.42916 .57207 L
.4299 .59428 L
.43071 .6146 L
s
.43071 .6146 m
.43089 .61803 L
s
.43512 .61803 m
.43628 .58981 L
.43752 .55055 L
.43886 .50624 L
.43956 .48546 L
.44032 .46803 L
.44159 .45721 L
.44278 .47327 L
.44414 .51128 L
.44538 .55259 L
.4466 .591 L
s
.4466 .591 m
.44771 .61803 L
s
.45194 .61803 m
.45223 .61206 L
.4529 .59492 L
.45543 .51324 L
.45655 .47997 L
.45776 .45846 L
.45907 .46314 L
.46029 .49093 L
.4625 .56299 L
.46367 .59845 L
s
.46367 .59845 m
.46454 .61803 L
s
.46874 .61803 m
.46915 .6094 L
.47056 .56897 L
.47174 .52912 L
.47302 .4885 L
.47375 .47026 L
.47442 .45924 L
.47574 .46061 L
.47643 .47303 L
.47718 .49268 L
.47869 .54191 L
.4802 .59048 L
.48103 .61219 L
s
.48103 .61219 m
.48133 .61803 L
s
.48553 .61803 m
.48614 .60418 L
.48744 .56593 L
.48873 .52245 L
.48995 .48443 L
.49063 .4683 L
.49137 .45742 L
.49266 .46188 L
.49388 .48959 L
.49519 .53131 L
.49641 .57205 L
.49753 .60444 L
s
.49753 .60444 m
.49821 .61803 L
s
.50223 .61803 m
.5028 .60719 L
.5056 .5199 L
.50636 .49543 L
.50717 .47361 L
.50786 .46061 L
.50863 .45501 L
.51 .47171 L
.51078 .49177 L
.51149 .51381 L
.51287 .56024 L
.51416 .59894 L
s
.51416 .59894 m
.51505 .61803 L
s
.519 .61803 m
.51939 .61106 L
.52066 .57674 L
.52182 .53872 L
.52307 .49777 L
.52444 .4638 L
.52563 .45514 L
.52675 .47009 L
.52793 .50166 L
.52919 .5431 L
.53056 .58654 L
.53131 .60636 L
s
.53131 .60636 m
.53187 .61803 L
s
.5358 .61803 m
.53652 .60197 L
.53802 .55794 L
.53942 .51142 L
.54069 .47482 L
.54204 .4549 L
.54332 .46521 L
.54467 .49909 L
.54612 .54646 L
.5474 .58612 L
.54875 .61802 L
s
.54875 .61802 m
.54875 .61803 L
s
.55249 .61803 m
.55254 .61745 L
.5538 .5878 L
.5553 .54113 L
.55666 .49733 L
.55795 .46507 L
.55917 .45479 L
.56046 .47195 L
.56119 .49038 L
.56185 .51045 L
.56304 .54896 L
.56431 .58731 L
.56551 .61498 L
s
.56551 .61498 m
.56574 .61803 L
s
.56914 .61803 m
.57025 .59545 L
.57161 .55597 L
.57288 .51488 L
.57424 .47586 L
.57502 .46085 L
.57574 .45489 L
.57646 .4584 L
.57714 .4693 L
.57785 .48604 L
.5786 .50786 L
.57995 .55085 L
.58142 .59311 L
.58225 .61123 L
s
.58225 .61123 m
.58269 .61803 L
s
.58581 .61803 m
.58586 .61745 L
.58724 .58825 L
.58854 .55048 L
.58978 .51101 L
.59096 .47791 L
.59231 .45618 L
.59355 .46239 L
.59587 .52146 L
.59803 .58559 L
.59912 .60934 L
s
.59912 .60934 m
.59981 .61803 L
s
.60227 .61803 m
.60275 .61299 L
.60402 .58638 L
.60541 .54682 L
.60663 .50891 L
.60792 .47426 L
.6091 .4568 L
.61039 .46326 L
.61157 .48832 L
.61267 .52005 L
.61407 .56232 L
.61541 .59596 L
.61661 .61571 L
s
.61661 .61571 m
.61703 .61803 L
s
.61882 .61803 m
.61933 .61266 L
.62011 .59963 L
.62084 .58329 L
.62216 .54667 L
.62355 .5047 L
.62477 .47347 L
.62611 .45658 L
.62733 .46587 L
.62847 .49053 L
.62977 .52776 L
.63119 .56835 L
.63254 .59864 L
.63378 .615 L
.63499 .61798 L
.63632 .60603 L
.63699 .59446 L
.63771 .57852 L
.63902 .54313 L
.64023 .50761 L
.64151 .47527 L
.6426 .45915 L
.6438 .46188 L
.64452 .47333 L
.64521 .48863 L
.64649 .52348 L
.64798 .56474 L
.64936 .5947 L
.65056 .61007 L
.65184 .61297 L
.65312 .60172 L
.65385 .58955 L
.65453 .575 L
.65589 .53926 L
.65738 .49785 L
.65806 .48128 L
.65879 .46752 L
.66009 .45906 L
.66133 .47392 L
.6625 .50059 L
.66516 .57025 L
.66589 .58516 L
.66656 .59607 L
.66726 .6039 L
.66803 .60794 L
.6693 .60379 L
.67065 .58524 L
.67142 .56942 L
.67225 .5491 L
.67376 .50871 L
Mistroke
.67447 .49075 L
.67523 .47473 L
.67658 .46018 L
.67785 .46979 L
.67923 .49802 L
.68044 .52912 L
.68174 .56109 L
.68303 .58585 L
.68419 .5991 L
.68536 .60195 L
.68646 .59474 L
.68916 .54388 L
.69063 .50647 L
.692 .47635 L
.69317 .46261 L
.69443 .468 L
.69514 .479 L
.69582 .49278 L
.69709 .52296 L
.69821 .54945 L
.69939 .57323 L
.7004 .58754 L
.70152 .59529 L
.70271 .59336 L
.704 .5798 L
.70521 .55784 L
.70633 .53256 L
.70745 .50564 L
.70865 .48039 L
.70997 .46473 L
.71119 .46908 L
.71257 .4914 L
.71387 .5201 L
.71515 .5482 L
.71631 .56944 L
.71752 .58395 L
.71882 .58885 L
.72003 .58295 L
.72117 .56873 L
.7225 .54404 L
.72373 .51692 L
.72442 .50173 L
.72517 .48682 L
.72652 .46861 L
.72722 .46644 L
.72795 .47055 L
.72927 .48941 L
.73175 .53989 L
.73313 .56352 L
.73441 .57741 L
Mistroke
.73515 .58097 L
.73582 .5812 L
.73659 .5779 L
.73733 .57143 L
.73965 .53381 L
.74183 .49052 L
.74293 .47474 L
.74411 .46929 L
.74542 .47994 L
.74661 .49912 L
.74917 .5458 L
.75046 .56311 L
.75118 .56945 L
.75186 .57275 L
.75309 .57202 L
.7542 .56401 L
.75549 .5473 L
.7567 .52652 L
.75792 .50412 L
.75922 .48333 L
.76046 .47268 L
.76158 .4754 L
.7629 .49046 L
.76433 .51365 L
.76568 .53558 L
.76694 .55223 L
.76823 .56263 L
.76892 .56485 L
.76966 .56455 L
.7704 .56149 L
.77117 .55548 L
.77256 .53871 L
.77507 .49893 L
.7758 .48871 L
.77648 .48113 L
.77776 .47576 L
.77896 .48271 L
.78006 .49565 L
.78125 .51262 L
.78255 .53098 L
.78367 .5439 L
.78472 .55217 L
.78583 .55602 L
.78702 .55418 L
.78832 .54545 L
.78905 .53786 L
.78973 .52953 L
.79108 .51095 L
.79231 .49462 L
.79352 .48285 L
Mistroke
.79465 .4796 L
.79585 .48565 L
.79715 .49877 L
.79965 .52806 L
.80036 .53496 L
.80104 .54021 L
.80232 .54603 L
.80352 .5462 L
.80463 .54179 L
.80585 .53253 L
.80714 .51908 L
.80844 .50424 L
.80912 .49694 L
.80983 .49038 L
.81115 .48378 L
.81235 .4863 L
.81304 .49052 L
.81378 .49645 L
.8151 .50886 L
.81666 .52294 L
.81735 .52802 L
.8181 .53243 L
.81882 .53531 L
.8195 .53673 L
.82021 .53685 L
.82097 .53539 L
.8218 .53205 L
.82259 .52738 L
.82407 .51573 L
.82538 .50418 L
.82676 .49349 L
.82756 .4895 L
.82829 .48817 L
.82901 .4891 L
.8297 .49155 L
.83098 .4988 L
.83234 .508 L
.83351 .51557 L
.83476 .52201 L
.83611 .52607 L
.83737 .52671 L
.83814 .52557 L
.83887 .52351 L
.84023 .51745 L
.84256 .50357 L
.84387 .49661 L
.84507 .49309 L
.84574 .49289 L
.84648 .49384 L
.84782 .49771 L
Mistroke
.8504 .50791 L
.85107 .51033 L
.8518 .51261 L
.85311 .51557 L
.8539 .5165 L
.85462 .51676 L
.856 .51571 L
.8572 .5133 L
.85852 .50942 L
.85989 .50467 L
.86119 .50026 L
.8624 .49744 L
.86372 .49633 L
.86492 .49663 L
.86602 .49769 L
.86722 .49943 L
.86854 .50175 L
.8712 .50663 L
.87233 .50837 L
.87352 .50972 L
.87483 .51039 L
.87603 .50999 L
.87715 .50847 L
.87819 .50581 L
.88055 .49795 L
.88189 .49407 L
.88257 .49255 L
.88331 .49132 L
.88462 .49042 L
.88585 .4912 L
.88659 .49243 L
.8873 .49409 L
.88862 .49829 L
.89006 .5039 L
.8916 .50982 L
.89295 .51353 L
.89419 .51417 L
.89542 .51076 L
.89659 .50461 L
.89923 .48838 L
.90056 .48239 L
.90132 .48029 L
.90203 .47932 L
.90326 .4801 L
.90439 .48352 L
.90568 .49028 L
.90691 .49862 L
.90804 .50687 L
.90909 .51398 L
.91019 .5193 L
Mistroke
.9114 .52058 L
.91262 .51513 L
.91393 .50414 L
.91626 .48192 L
.91761 .47236 L
.91828 .46926 L
.91902 .46739 L
.92032 .4683 L
.92154 .47392 L
.92289 .48472 L
.92431 .49952 L
.92554 .51274 L
.92686 .52405 L
.9276 .52757 L
.92827 .52806 L
.9296 .52023 L
.93034 .5122 L
.93115 .50188 L
.93261 .48223 L
.93397 .46681 L
.93524 .45742 L
.93657 .45453 L
.93727 .4561 L
.93803 .46017 L
.93931 .47196 L
.94065 .48949 L
.94191 .50792 L
.94306 .52331 L
.94425 .53421 L
.94539 .53537 L
.94658 .52482 L
.94788 .50505 L
.95002 .46893 L
.95124 .45298 L
.95238 .44389 L
.95362 .44179 L
.95495 .44901 L
.95568 .45684 L
.95638 .46645 L
.95768 .48863 L
.95882 .50973 L
.96003 .53005 L
.96135 .54369 L
.96258 .54143 L
.96491 .50281 L
.96709 .45821 L
.96818 .44128 L
.96935 .43051 L
.97065 .42909 L
.97184 .43773 L
Mistroke
.97312 .45664 L
.97452 .48529 L
.97575 .51315 L
.97706 .53916 L
.97824 .55286 L
.97955 .54856 L
.98074 .52799 L
.98185 .50125 L
.98407 .44745 L
.98519 .42788 L
.98641 .41624 L
.98746 .4156 L
.98857 .42463 L
.98963 .44159 L
.99059 .46273 L
.99291 .52381 L
.99422 .55222 L
.99543 .56374 L
.99649 .56146 L
.99765 .55861 L
1 .5521 L
Mfstroke
0 0 1 r
0 .30902 m
.00123 .43354 L
.00236 .44334 L
.00365 .46234 L
.00488 .47993 L
.00601 .48718 L
.00706 .48678 L
.00821 .48099 L
.00942 .47603 L
.01179 .46414 L
.01312 .45934 L
.01434 .45536 L
.01502 .45472 L
.01577 .45576 L
.01713 .4536 L
.01849 .45434 L
.01975 .45511 L
.02494 .45901 L
.02617 .46082 L
.02748 .46147 L
.0286 .46183 L
.02982 .46202 L
.0305 .46205 L
.03122 .46203 L
.03253 .46178 L
.03332 .46152 L
.03406 .46123 L
.03474 .46097 L
.03546 .46071 L
.03673 .4604 L
.03791 .46045 L
.03917 .46006 L
.03989 .45976 L
.04057 .45951 L
.0419 .45915 L
.0426 .45908 L
.04336 .45913 L
.04408 .45934 L
.04487 .45949 L
.04561 .45937 L
.0463 .45927 L
.04763 .45912 L
.04831 .45909 L
.04905 .45909 L
.05034 .45929 L
.05155 .45944 L
.05223 .45941 L
.05295 .45938 L
.05374 .45937 L
.05448 .45941 L
Mistroke
.05523 .4595 L
.05593 .45966 L
.05726 .46024 L
.05806 .4602 L
.0588 .4601 L
.0596 .45993 L
.06046 .45971 L
.06199 .4592 L
.06339 .45873 L
.0647 .45839 L
.06594 .45831 L
.06725 .45869 L
.06799 .4592 L
.06867 .45992 L
.06981 .46179 L
.07107 .46169 L
.07238 .46055 L
.07361 .45939 L
.07498 .45817 L
.07573 .45761 L
.07642 .4572 L
.07765 .4569 L
.079 .45746 L
.08481 .45936 L
.08615 .45811 L
.08739 .45738 L
.0887 .45741 L
.08939 .45788 L
.09013 .45883 L
.09093 .46049 L
.0917 .46278 L
.09238 .46552 L
.09313 .46935 L
.09382 .46897 L
.09456 .46836 L
.09589 .46708 L
.09827 .46468 L
.09962 .46369 L
.10085 .46337 L
.10223 .4641 L
.10292 .46505 L
.10368 .46667 L
.10505 .46921 L
.10633 .46693 L
.10887 .4619 L
.11012 .4596 L
.11127 .45788 L
.11232 .45688 L
.11343 .45666 L
.11454 .45762 L
Mistroke
.11577 .46047 L
.11708 .46009 L
.11828 .45754 L
.11957 .45482 L
.12098 .45216 L
.12228 .45032 L
.12348 .44946 L
.12463 .44973 L
.12586 .45161 L
.12655 .45354 L
.12728 .45644 L
.1286 .45733 L
.12998 .45528 L
.13149 .45298 L
.13288 .45107 L
.1342 .44973 L
.13539 .44921 L
.13667 .4497 L
.13739 .45063 L
.13807 .45203 L
.13888 .45446 L
.13962 .45556 L
.14035 .45514 L
.14112 .45465 L
.14274 .45343 L
.14543 .45081 L
.14836 .44696 L
.15109 .44252 L
.15367 .43906 L
.15509 .4374 L
.15584 .43664 L
.15664 .43596 L
.15731 .43549 L
.15804 .43511 L
.15934 .43484 L
.16067 .43521 L
.16188 .43625 L
.16319 .43715 L
.16388 .43757 L
.16461 .43813 L
.16592 .43942 L
.16714 .44096 L
.16954 .44492 L
.17227 .45087 L
.17482 .45675 L
.1761 .45894 L
.17747 .46078 L
.17826 .46151 L
.179 .46193 L
.18038 .46187 L
Mistroke
.18112 .46129 L
.1819 .46019 L
.18257 .45881 L
.18331 .45676 L
.18648 .44505 L
.18941 .43711 L
.19208 .43031 L
.19338 .42781 L
.19411 .42678 L
.19479 .42614 L
.19603 .42593 L
.19735 .42612 L
.19809 .42568 L
.19889 .42548 L
.19964 .42557 L
.20033 .42593 L
.20164 .4274 L
.20303 .43026 L
.20425 .43403 L
.20555 .43956 L
.20679 .44635 L
.20791 .45402 L
.21067 .4682 L
.21193 .47356 L
.21328 .47842 L
.21458 .48169 L
.21576 .48305 L
.2169 .48243 L
.21809 .47927 L
.21941 .47212 L
.22063 .46674 L
.22206 .4622 L
.22341 .45672 L
.22614 .44242 L
.22861 .42651 L
.23132 .40768 L
.23279 .39926 L
.23418 .39301 L
.23528 .3896 L
.23647 .38802 L
.23779 .38948 L
.239 .39448 L
.24009 .40264 L
.24128 .41617 L
.2424 .43399 L
.24345 .44899 L
.24478 .46233 L
.24598 .47487 L
.24867 .50173 L
.2498 .51112 L
Mistroke
.251 .51873 L
.25231 .52328 L
.25351 .52279 L
.2548 .51964 L
.25552 .52085 L
.25619 .52125 L
.25743 .52002 L
.25873 .51557 L
.2599 .5085 L
.26099 .4989 L
.26219 .48473 L
.26349 .4648 L
.2662 .42198 L
.26757 .40683 L
.26835 .39947 L
.26906 .39396 L
.27034 .38778 L
.27171 .38825 L
.27297 .39729 L
.27417 .41544 L
.27537 .44071 L
.2765 .44966 L
.27768 .45969 L
.27898 .47071 L
.28008 .4792 L
.28112 .48571 L
.28232 .49025 L
.28341 .4904 L
.28462 .48457 L
.2859 .47519 L
.287 .47511 L
.28821 .47343 L
.2895 .46955 L
.29023 .46633 L
.2909 .46269 L
.29213 .45415 L
.29343 .44245 L
.29612 .41531 L
.29744 .40459 L
.29866 .39623 L
.29977 .39075 L
.30095 .38811 L
.30213 .38988 L
.30342 .39851 L
.30858 .43261 L
.31142 .45623 L
.31412 .47761 L
.31532 .48587 L
.3166 .49211 L
.31734 .49405 L
Mistroke
.31801 .49451 L
.31932 .49102 L
.32049 .48176 L
.32176 .46358 L
.32296 .43889 L
.32407 .42723 L
.3252 .41554 L
.32643 .40399 L
.32764 .39552 L
.32893 .39149 L
.33013 .39462 L
.33126 .4059 L
.33237 .42306 L
.33339 .42659 L
.33459 .43174 L
.33588 .43801 L
.33709 .44394 L
.33821 .4488 L
.33945 .45246 L
.3408 .45427 L
.34212 .45987 L
.34286 .46254 L
.34353 .46459 L
.34421 .46616 L
.34493 .46718 L
.34565 .46741 L
.34644 .46657 L
.34712 .46479 L
.34785 .46161 L
.34918 .45327 L
.35055 .44935 L
.35186 .44578 L
.35303 .44333 L
.3543 .44237 L
.35562 .44439 L
.35686 .45036 L
.35797 .45507 L
.35917 .4566 L
.36048 .45824 L
.36186 .45953 L
.36305 .45983 L
.36434 .45873 L
.36554 .45578 L
.36667 .4548 L
.36797 .45462 L
.36916 .45446 L
.37028 .45437 L
.37131 .45439 L
.37244 .45459 L
.37365 .4551 L
Mistroke
.37491 .4555 L
.37629 .4556 L
.3788 .45579 L
.38352 .45608 L
.39414 .45688 L
.41335 .45815 L
.41583 .4585 L
.41708 .45865 L
.41845 .45875 L
.41917 .45877 L
.41994 .45876 L
.42132 .45863 L
.42271 .45833 L
.42397 .45784 L
.42525 .45765 L
.42643 .45788 L
.42911 .45847 L
.43162 .45903 L
.43286 .45926 L
.43399 .45942 L
.43526 .4595 L
.43664 .45943 L
.43733 .45931 L
.43807 .4591 L
.43885 .45879 L
.43958 .45841 L
.44029 .45792 L
.44106 .45725 L
.44178 .45705 L
.44245 .4572 L
.44406 .45764 L
.44553 .45808 L
.44815 .4589 L
.44944 .45924 L
.45061 .45948 L
.45185 .45962 L
.45318 .45958 L
.45387 .45946 L
.45462 .45923 L
.456 .4585 L
.45731 .45736 L
.45868 .4565 L
.45941 .45673 L
.4602 .45701 L
.46159 .45754 L
.46413 .45856 L
.46553 .45908 L
.46685 .45948 L
.46812 .45973 L
.46951 .45978 L
Mistroke
.47077 .45955 L
.47193 .45903 L
.47422 .45685 L
.4755 .45593 L
.47668 .45634 L
.4793 .45741 L
.48176 .45845 L
.48317 .45896 L
.48449 .45931 L
.48586 .45947 L
.48659 .45945 L
.48736 .45931 L
.48813 .45904 L
.48883 .45867 L
.4896 .45809 L
.49042 .45727 L
.4918 .45541 L
.49258 .4557 L
.49331 .45599 L
.49611 .45725 L
.49871 .45846 L
.50001 .45897 L
.50124 .45933 L
.50193 .45945 L
.50266 .45951 L
.50394 .45939 L
.50463 .45917 L
.50539 .45879 L
.50618 .45821 L
.50692 .45748 L
.50759 .45663 L
.50832 .45548 L
.50965 .45552 L
.51101 .45612 L
.51227 .45673 L
.51466 .45792 L
.51601 .45853 L
.51727 .459 L
.51858 .45933 L
.51928 .45941 L
.52003 .4594 L
.52132 .45911 L
.52252 .45844 L
.52384 .45715 L
.52457 .45613 L
.52524 .45498 L
.52647 .45527 L
.52779 .45589 L
.5302 .45715 L
.53255 .45836 L
Mistroke
.53384 .45891 L
.53504 .45928 L
.53607 .45945 L
.53717 .45941 L
.53839 .45903 L
.5395 .45831 L
.54075 .45696 L
.5421 .4547 L
.54338 .45521 L
.54455 .45581 L
.54709 .45725 L
.54855 .45808 L
.54991 .45878 L
.55099 .45924 L
.55215 .45959 L
.55343 .45971 L
.5546 .45952 L
.55582 .45892 L
.55714 .45766 L
.55829 .45594 L
.55954 .45499 L
.56089 .4557 L
.56216 .45643 L
.56504 .4582 L
.56661 .45908 L
.56738 .45945 L
.56809 .45974 L
.56942 .46009 L
.57013 .46014 L
.57088 .46008 L
.57224 .45957 L
.573 .45902 L
.5737 .4583 L
.57497 .45644 L
.57569 .455 L
.57636 .45525 L
.57753 .4559 L
.57878 .45666 L
.58133 .45832 L
.58273 .45919 L
.58403 .4599 L
.5854 .46045 L
.58612 .46063 L
.5869 .46071 L
.58821 .46052 L
.58958 .45974 L
.59076 .45848 L
.59204 .45634 L
.59283 .45545 L
.59357 .45586 L
Mistroke
.59496 .45672 L
.59807 .45884 L
.59938 .45971 L
.60076 .46052 L
.60148 .46087 L
.60228 .46117 L
.603 .46135 L
.60367 .46143 L
.60498 .46124 L
.60572 .4609 L
.60641 .46041 L
.60767 .45898 L
.60901 .45655 L
.6102 .45633 L
.61132 .45704 L
.61387 .45885 L
.61643 .46067 L
.61783 .46151 L
.61913 .46208 L
.61982 .46226 L
.62055 .46235 L
.62136 .46228 L
.62209 .46205 L
.62276 .46167 L
.62349 .46105 L
.62481 .45929 L
.6261 .45661 L
.62728 .4573 L
.62993 .45922 L
.63242 .46113 L
.63363 .46199 L
.63473 .46266 L
.63603 .46325 L
.63741 .46351 L
.63813 .46345 L
.63891 .4632 L
.63965 .46276 L
.64033 .46215 L
.64157 .46046 L
.64292 .45759 L
.6436 .45803 L
.64433 .45853 L
.64565 .45951 L
.6485 .46181 L
.65 .46298 L
.65082 .46355 L
.65159 .46403 L
.65298 .46468 L
.65371 .46486 L
.65451 .46491 L
Mistroke
.65519 .4648 L
.65592 .46451 L
.65721 .46345 L
.65847 .46161 L
.65981 .45891 L
.66108 .4598 L
.66224 .4607 L
.66486 .46291 L
.66625 .46407 L
.66774 .4652 L
.66911 .46602 L
.67037 .46647 L
.67163 .46653 L
.67283 .4661 L
.67417 .46489 L
.67541 .46291 L
.67678 .46054 L
.67754 .4611 L
.67823 .46163 L
.68068 .46374 L
.68327 .46602 L
.68433 .46687 L
.68547 .46764 L
.68673 .46824 L
.68789 .46847 L
.68859 .46842 L
.68935 .46817 L
.69073 .46708 L
.69207 .46505 L
.69331 .46215 L
.69465 .46319 L
.69614 .46447 L
.69915 .46726 L
.7005 .46846 L
.70178 .46946 L
.70249 .46991 L
.70326 .47031 L
.70461 .47067 L
.7059 .47049 L
.70727 .46956 L
.70856 .46778 L
.70974 .46523 L
.71099 .46496 L
.71217 .46597 L
.71485 .46852 L
.71767 .47117 L
.71891 .47212 L
.72025 .47288 L
.7215 .47319 L
.72265 .47303 L
Mistroke
.72375 .47238 L
.72492 .471 L
.72615 .46864 L
.7273 .46701 L
.72857 .46811 L
.72993 .46941 L
.73281 .47232 L
.73421 .47367 L
.73548 .47473 L
.73683 .47558 L
.73757 .47588 L
.73827 .47602 L
.73904 .47599 L
.73974 .47577 L
.74051 .47526 L
.74133 .4744 L
.74257 .4723 L
.7439 .46964 L
.74516 .47076 L
.74631 .47188 L
.74918 .47488 L
.75043 .47617 L
.75179 .47743 L
.75313 .47844 L
.75437 .47904 L
.75555 .47922 L
.7568 .47883 L
.75812 .4776 L
.75935 .47551 L
.7607 .47279 L
.76145 .47347 L
.76214 .47413 L
.76456 .47667 L
.76713 .47946 L
.76933 .48149 L
.77058 .4823 L
.77175 .48271 L
.7731 .48262 L
.77433 .48183 L
.77568 .47998 L
.77636 .47858 L
.77711 .47663 L
.77787 .47667 L
.77857 .47734 L
.78013 .47897 L
.78293 .48212 L
.78555 .48483 L
.78626 .48543 L
.78702 .48597 L
.78838 .48658 L
Mistroke
.78965 .48661 L
.79083 .48603 L
.79213 .48452 L
.79285 .48319 L
.79352 .48163 L
.79475 .48069 L
.79591 .48187 L
.79855 .48486 L
.80134 .48803 L
.80256 .48923 L
.8039 .49028 L
.80507 .49084 L
.80615 .49097 L
.8073 .49059 L
.80853 .48943 L
.80987 .48706 L
.81114 .48464 L
.81228 .48581 L
.81352 .48719 L
.81636 .49056 L
.81902 .49353 L
.82035 .49469 L
.82111 .49518 L
.82179 .4955 L
.82254 .49568 L
.82336 .49563 L
.82413 .49531 L
.82485 .49474 L
.8261 .49299 L
.82745 .4898 L
.82873 .49025 L
.82991 .49156 L
.83274 .49499 L
.8353 .49802 L
.83663 .49933 L
.83788 .50026 L
.83901 .50074 L
.84024 .50076 L
.84157 .49999 L
.84224 .49922 L
.84299 .49802 L
.8443 .49488 L
.84553 .4955 L
.84669 .49684 L
.84792 .49833 L
.85012 .5011 L
.85146 .50272 L
.85271 .50412 L
.85394 .50525 L
.85507 .506 L
Mistroke
.85632 .50638 L
.85767 .5061 L
.85901 .50488 L
.85969 .50383 L
.86044 .50229 L
.86168 .50051 L
.86302 .50202 L
.86543 .50502 L
.86847 .50886 L
.86997 .51049 L
.87136 .51166 L
.87261 .51228 L
.87395 .51234 L
.87471 .51201 L
.87543 .51141 L
.87677 .50942 L
.87799 .5064 L
.87931 .50762 L
.88167 .51056 L
.88425 .51392 L
.88561 .5156 L
.88705 .51714 L
.88777 .51777 L
.88857 .51833 L
.88929 .51869 L
.88997 .51885 L
.8913 .51863 L
.89204 .51812 L
.89273 .51737 L
.89393 .51526 L
.89525 .51331 L
.89662 .51492 L
.89789 .51654 L
.90022 .51961 L
.90273 .52279 L
.90388 .52402 L
.9051 .52505 L
.90614 .52561 L
.90729 .5258 L
.9086 .52531 L
.90933 .52463 L
.91 .52371 L
.91122 .52121 L
.91254 .52105 L
.91381 .52261 L
.91499 .52414 L
.9173 .52726 L
.91967 .53025 L
.92096 .53161 L
.92216 .53257 L
Mistroke
.92346 .53316 L
.92415 .53322 L
.92489 .53305 L
.9262 .53206 L
.9274 .53021 L
.92861 .52788 L
.92974 .52921 L
.93227 .53252 L
.93449 .53553 L
.93688 .5385 L
.93823 .53985 L
.93946 .54071 L
.9407 .54111 L
.94188 .54091 L
.94324 .53978 L
.94451 .53766 L
.94524 .53588 L
.9459 .53664 L
.94738 .53849 L
.95279 .54573 L
.95411 .54725 L
.95551 .54855 L
.95632 .54908 L
.95706 .54939 L
.9578 .5495 L
.9585 .5494 L
.95976 .5486 L
.96109 .54672 L
.96223 .54481 L
.96348 .54633 L
.96609 .54978 L
.96889 .5536 L
.97014 .55517 L
.97146 .5566 L
.97259 .55756 L
.97382 .55822 L
.97498 .55837 L
.97622 .55788 L
.97738 .55665 L
.97844 .55474 L
.97979 .55498 L
.98047 .55583 L
.98121 .55679 L
.98377 .56026 L
.98655 .56395 L
.98808 .56569 L
.98884 .56641 L
.98953 .56696 L
.99089 .56765 L
.9916 .56776 L
Mistroke
.99236 .56766 L
.99372 .56677 L
.99494 .56504 L
.9962 .56283 L
.99755 .56022 L
.99883 .5566 L
1 .5521 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[46]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`004`3oool5000000030?ooo`00
0000oooo0080oooo00<000000?ooo`3oool00P3oool500000080oooo0P00003e0?ooo`005@3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0oooo0000
00020?ooo`030000003oool0oooo0?80oooo000E0?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool01P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0lP3o
ool000d0oooo100000040?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000001`3oool0
1@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3oool3000001/0oooo0P3<
c<`50?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0
oooo0?ooo`3<c<`30?ooo`030<c<c03oool0oooo0080oooo00@0c<c<0?ooo`3oool0c<c<103oool0
0`3<c<`0oooo0<c<c00;0?ooo`030<c<c03oool0oooo00T0oooo0P3<c<`:0?ooo`030000003<c<`0
000007T0oooo000E0?ooo`050000003oool0oooo0?ooo`0000000`3oool00`000000oooo0?ooo`04
0?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`070?ooo`030000003o
ool0oooo01/0oooo0P3<c<`50?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0
c<c<103oool0103<c<`0oooo0?ooo`3<c<`30?ooo`030<c<c03oool0oooo0080oooo00@0c<c<0?oo
o`3oool0c<c<103oool00`3<c<`0oooo0<c<c00;0?ooo`030<c<c03oool0oooo00T0oooo0P3<c<`9
0?ooo`@00000N@3oool001@0oooo0P00000=0?ooo`800000103oool2000000T0oooo0P00000L0?oo
o`030<c<c03oool0c<c<00D0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0oooo0?ooo`3<
c<`40?ooo`040<c<c03oool0oooo0<c<c0<0oooo00<0c<c<0?ooo`3oool00P3oool0103<c<`0oooo
0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00/0oooo00<0c<c<0?ooo`3oool02@3oool20<c<c0T0
oooo1000001i0?ooo`00=03oool3000001/0oooo00<0c<c<0?ooo`3<c<`01@3oool0103<c<`0oooo
0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<0`3oool0
0`3<c<`0oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0@0oooo00<0c<c<0?ooo`3<c<`02P3o
ool00`000000c<c<00000003000000L0oooo0P3<c<`90?ooo`<00000NP3oool002h0oooo0`000004
0?ooo`030000003oool0oooo01X0oooo00<0c<c<0?ooo`3<c<`01@3oool0103<c<`0oooo0?ooo`3<
c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<0`3oool00`3<c<`0
oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0@0oooo00<0c<c<0?ooo`3<c<`02@3oool40000
00<0oooo00<000000?ooo`3oool0103oool20<c<c0P0oooo00<000000?ooo`3oool0N`3oool003D0
oooo00<000000?ooo`3oool06P3oool00`3<c<`0oooo0<c<c0050?ooo`040<c<c03oool0oooo0<c<
c0@0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0oooo0?ooo`3<c<`30?ooo`030<c<c03o
ool0oooo0080oooo00@0c<c<0?ooo`3oool0c<c<103oool00`3<c<`0oooo0<c<c0090?ooo`@00000
103oool00`000000oooo0?ooo`040?ooo`040<c<c03oool0oooo0?ooo`D00000OP3oool003<0oooo
0`00000L0?ooo`030<c<c03oool0c<c<00D0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0
oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0<0oooo00<0c<c<0?ooo`3oool00P3oool0
103<c<`0oooo0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00X0oooo0P0000060?ooo`800000103o
ool0103<c<`0oooo0000000000230?ooo`00=@3oool2000001/0oooo00<0c<c<0?ooo`3<c<`01@3o
ool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3o
ool0c<c<0`3oool00`3<c<`0oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0@0oooo00<0c<c<
0?ooo`3<c<`02@3oool00`000000oooo0?ooo`080?ooo`80000000@0oooo0000003<c<`00000Q@3o
ool003D0oooo00<000000?ooo`3oool06P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03o
ool0oooo0<c<c0@0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0030?ooo`040<c<c03oool0oooo0<c<
c0<0oooo00<0c<c<0?ooo`3oool00P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`030<c<c03oool0
c<c<00T0oooo00<000000?ooo`3oool02@3oool5000008D0oooo000e0?ooo`030000003oool0oooo
01X0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0oooo0?ooo`3<c<`40?ooo`050<c<c03o
ool0oooo0?ooo`3<c<`00`3oool01@3<c<`0oooo0?ooo`3oool0c<c<0080oooo00<0c<c<0?ooo`3o
ool00P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00T0oooo00<000000?oo
o`3oool02P3oool4000008D0oooo000e0?ooo`030000003oool0oooo01X0oooo00@0c<c<0?ooo`3o
ool0c<c<103oool0103<c<`0oooo0?ooo`3<c<`40?ooo`050<c<c03oool0oooo0?ooo`3<c<`00`3o
ool01@3<c<`0oooo0?ooo`3oool0c<c<0080oooo00<0c<c<0?ooo`3oool00P3oool0103<c<`0oooo
0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00T0oooo00<000000?ooo`3oool02`3oool2000008H0
oooo000e0?ooo`8000006`3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<
c0@0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0030?ooo`050<c<c03oool0oooo0?ooo`3<c<`00P3o
ool00`3<c<`0oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0@0oooo00<0c<c<0?ooo`3<c<`0
203oool00`000000oooo0?ooo`2D0?ooo`00=@3oool00`000000oooo0?ooo`0J0?ooo`040<c<c03o
ool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<103oool01@3<c<`0oooo0?ooo`3oool0c<c<
00<0oooo00D0c<c<0?ooo`3oool0oooo0<c<c0020?ooo`030<c<c03oool0oooo0080oooo00@0c<c<
0?ooo`3oool0c<c<103oool00`3<c<`0oooo0<c<c0080?ooo`030000003oool0oooo09@0oooo000e
0?ooo`030000003oool0oooo01X0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0oooo0?oo
o`3<c<`30?ooo`030<c<c03oool0oooo0080oooo00D0c<c<0?ooo`3oool0oooo0<c<c0030?ooo`04
0<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<1@3oool20<c<c0P0oooo00<00000
0?ooo`3oool0U03oool000T0oooo00<000000?ooo`3oool00P3oool2000000@0oooo0P0000030?oo
o`050000003oool0oooo0?ooo`000000103oool500000080oooo0P00000:0?ooo`8000006`3oool0
103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0<0oooo00<0c<c<0?ooo`3oool0
0P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?oo
o`3oool0c<c<1@3oool20<c<c0L0oooo00<000000?ooo`3oool0U@3oool000T0oooo00<000000?oo
o`3oool01`3oool010000000oooo0?ooo`0000030?ooo`030000003oool0000000L0oooo00D00000
0?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool06P3o
ool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0<0oooo00<0c<c<0?ooo`3o
ool00P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<
0?ooo`3oool0c<c<1@3oool20<c<c0L0oooo00<000000?ooo`3oool0U@3oool000T0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00H0oooo00D00000
0?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool06P3o
ool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0<0oooo00<0c<c<0?ooo`3o
ool00P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<
0?ooo`3oool0c<c<1@3oool20<c<c0L0oooo00<000000?ooo`3oool0U@3oool000030?ooo`000000
0000008000001@3oool00`000000oooo0?ooo`070?ooo`800000103oool00`000000oooo00000007
0?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`070?ooo`030000003o
ool0oooo01X0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0oooo0?ooo`3<c<`30?ooo`03
0<c<c03oool0oooo0080oooo00@0c<c<0?ooo`3oool0c<c<103oool00`3<c<`0oooo0<c<c0050?oo
o`040<c<c03oool0oooo0<c<c0D0oooo0P3<c<`60?ooo`030000003oool0oooo09H0oooo00070?oo
o`040000003oool0oooo000000T0oooo00<000000?ooo`3oool00P3oool01@000000oooo0?ooo`3o
ool0000000H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo
0`00000J0?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<0`3oool00`3<
c<`0oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0@0oooo00<0c<c<0?ooo`3<c<`01@3oool0
103<c<`0oooo0?ooo`3<c<`50?ooo`80c<c<1P3oool00`000000oooo0?ooo`2F0?ooo`001`3oool4
000000T0oooo0`00000<0?ooo`800000103oool2000000X0oooo00<000000?ooo`3oool06@3oool0
0`3<c<`0oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0<0oooo00D0c<c<0?ooo`3oool0oooo
0<c<c0040?ooo`040<c<c03oool0oooo0<c<c0@0oooo00<0c<c<0?ooo`3<c<`01@3oool0103<c<`0
oooo0?ooo`3<c<`50?ooo`80c<c<1P3oool00`000000oooo0?ooo`2F0?ooo`00<`3oool3000001/0
oooo00<0c<c<0?ooo`3oool00P3oool0103<c<`0oooo0?ooo`3<c<`30?ooo`050<c<c03oool0oooo
0?ooo`3<c<`0103oool0103<c<`0oooo0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00D0oooo00@0
c<c<0?ooo`3oool0c<c<1@3oool20<c<c0D0oooo00<000000?ooo`3oool0U`3oool003<0oooo00@0
00000?ooo`00000000006P3oool00`3<c<`0oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0<0
oooo00D0c<c<0?ooo`3oool0oooo0<c<c0040?ooo`040<c<c03oool0oooo0<c<c0@0oooo00<0c<c<
0?ooo`3<c<`01@3oool0103<c<`0oooo0?ooo`3<c<`50?ooo`80c<c<1@3oool00`000000oooo0?oo
o`2G0?ooo`00;P3oool3000000<0oooo0P00000K0?ooo`030<c<c03oool0oooo0080oooo00@0c<c<
0?ooo`3oool0c<c<0`3oool01@3<c<`0oooo0?ooo`3oool0c<c<00@0oooo00@0c<c<0?ooo`3oool0
c<c<103oool00`3<c<`0oooo0<c<c0060?ooo`030<c<c03oool0c<c<00D0oooo0P3<c<`50?ooo`03
0000003oool0oooo09L0oooo000c0?ooo`030000003oool0000001/0oooo00<0c<c<0?ooo`3oool0
0P3oool0103<c<`0oooo0?ooo`3<c<`30?ooo`050<c<c03oool0oooo0?ooo`3<c<`0103oool0103<
c<`0oooo0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00H0oooo00<0c<c<0?ooo`3<c<`01@3oool2
0<c<c0D0oooo00<000000?ooo`3oool0U`3oool003<0oooo0`00000K0?ooo`030<c<c03oool0oooo
0080oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03o
ool0oooo0<c<c0@0oooo00<0c<c<0?ooo`3<c<`01P3oool00`3<c<`0oooo0<c<c0050?ooo`80c<c<
103oool00`000000oooo0?ooo`2H0?ooo`00=@3oool2000001X0oooo00<0c<c<0?ooo`3oool00P3o
ool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3o
ool0c<c<103oool00`3<c<`0oooo0<c<c0060?ooo`030<c<c03oool0c<c<00D0oooo0P3<c<`40?oo
o`030000003oool0oooo09P0oooo000e0?ooo`030000003oool0oooo01T0oooo00<0c<c<0?ooo`3o
ool00P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<
0?ooo`3oool0c<c<103oool00`3<c<`0oooo0<c<c0060?ooo`030<c<c03oool0c<c<00D0oooo0P3<
c<`40?ooo`030000003oool0oooo09P0oooo000e0?ooo`030000003oool0oooo01T0oooo00<0c<c<
0?ooo`3oool00P3oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo
00@0c<c<0?ooo`3oool0c<c<103oool00`3<c<`0oooo0<c<c0060?ooo`030<c<c03oool0c<c<00D0
oooo0P3<c<`40?ooo`030000003oool0oooo09P0oooo000e0?ooo`8000006P3oool00`3<c<`0oooo
0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<
c<`0oooo0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`60?ooo`80c<c<0`3oool0
0`000000oooo0?ooo`2I0?ooo`00=@3oool00`000000oooo0?ooo`0I0?ooo`030<c<c03oool0oooo
0080oooo00@0c<c<0?ooo`3oool0c<c<103oool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03o
ool0oooo0<c<c0@0oooo00<0c<c<0?ooo`3<c<`01P3oool20<c<c0H0oooo0P3<c<`30?ooo`030000
003oool0oooo09T0oooo000e0?ooo`030000003oool0oooo01T0oooo00<0c<c<0?ooo`3oool00P3o
ool0103<c<`0oooo0?ooo`3<c<`40?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3o
ool0c<c<1@3oool20<c<c0H0oooo0P3<c<`60?ooo`80c<c<0`3oool00`000000oooo0?ooo`2I0?oo
o`00=@3oool00`000000oooo0?ooo`0I0?ooo`030<c<c03oool0oooo0080oooo00@0c<c<0?ooo`3o
ool0c<c<103oool0103<c<`0oooo0?ooo`3<c<`40?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`6
0?ooo`80c<c<1P3oool20<c<c0<0oooo00<000000?ooo`3oool0V@3oool001@0oooo0P0000030?oo
o`050000003oool0oooo0?ooo`000000103oool500000080oooo0P00000:0?ooo`8000006P3oool0
0`3<c<`0oooo0?ooo`020?ooo`040<c<c03oool0oooo0<c<c0@0oooo00@0c<c<0?ooo`3oool0c<c<
103oool00`3<c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<c0H0oooo0P3<c<`20?ooo`030000
003oool0oooo09X0oooo000C0?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`000000
1`3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3oool00`000000
oooo0?ooo`0I0?ooo`030<c<c03oool0oooo0080oooo00@0c<c<0?ooo`3oool0c<c<103oool00`3<
c<`0oooo0<c<c0050?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`60?ooo`80c<c<1P3oool20<c<
c080oooo00<000000?ooo`3oool0VP3oool001H0oooo00<000000?ooo`3oool00P3oool00`000000
oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo01T0oooo00<0c<c<0?ooo`3oool00P3oool0103<c<`0oooo0?ooo`3<c<`4
0?ooo`030<c<c03oool0c<c<00D0oooo00<0c<c<0?ooo`3<c<`01P3oool20<c<c0H0oooo0P3<c<`6
0?ooo`80c<c<00<0oooo0000003oool0W03oool000d0oooo100000030?ooo`800000103oool00`00
0000oooo000000070?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`07
0?ooo`<000006@3oool00`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0c<c<00D0oooo00<0c<c<
0?ooo`3<c<`01@3oool00`3<c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<c0H0oooo0P3<c<`0
0`000000oooo0?ooo`2L0?ooo`00503oool00`000000oooo0?ooo`020?ooo`050000003oool0oooo
0?ooo`0000001P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3o
ool00`000000oooo0?ooo`0I0?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3<c<`01@3o
ool00`3<c<`0oooo0<c<c0050?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`60?ooo`80c<c<1@3o
ool4000009h0oooo000D0?ooo`<00000303oool2000000@0oooo0P0000080?ooo`<000006`3oool0
0`3<c<`0oooo0?ooo`020?ooo`030<c<c03oool0c<c<00D0oooo00<0c<c<0?ooo`3<c<`01@3oool0
0`3<c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<c0D0oooo1000002N0?ooo`00<`3oool00`00
0000oooo0000000K0?ooo`030<c<c03oool0oooo0080oooo00<0c<c<0?ooo`3<c<`01@3oool00`3<
c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<c0H0oooo0P3<c<`40?ooo`040000003oool00000
000009l0oooo000^0?ooo`<000000`3oool3000001X0oooo00<0c<c<0?ooo`3oool00P3oool00`3<
c<`0oooo0<c<c0050?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`60?ooo`80c<c<1P3oool20<c<
c0@0oooo00<000000?ooo`3oool0X03oool003<0oooo00<000000?ooo`0000006`3oool00`3<c<`0
oooo0?ooo`020?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`60?ooo`80c<c<1P3oool20<c<c0H0
oooo0P3<c<`30?ooo`030000003oool0oooo0:40oooo000c0?ooo`<000006`3oool00`3<c<`0oooo
0?ooo`020?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`60?ooo`80c<c<1P3oool20<c<c0H0oooo
0P3<c<`300000:@0oooo000e0?ooo`030000003oool0oooo01P0oooo00<0c<c<0?ooo`3oool00`3o
ool00`3<c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<c0H0oooo0P3<c<`50?ooo`@00000YP3o
ool003D0oooo0P00000I0?ooo`030<c<c03oool0oooo00<0oooo00<0c<c<0?ooo`3<c<`01P3oool2
0<c<c0H0oooo0P3<c<`60?ooo`80c<c<1@3oool400000:H0oooo000e0?ooo`030000003oool0oooo
01P0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<
c0H0oooo0P3<c<`40?ooo`040000003oool0000000000:L0oooo000e0?ooo`030000003oool0oooo
01P0oooo00<0c<c<0?ooo`3oool00`3oool00`3<c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<
c0H0oooo0P3<c<`30?ooo`030000003oool0oooo0:T0oooo000e0?ooo`8000006@3oool00`3<c<`0
oooo0?ooo`030?ooo`030<c<c03oool0c<c<00H0oooo0P3<c<`60?ooo`80c<c<1P3oool20<c<c003
0?ooo`00000000000:`0oooo000e0?ooo`030000003oool0oooo01P0oooo00<0c<c<0?ooo`3oool0
0`3oool00`3<c<`0oooo0<c<c0060?ooo`80c<c<1P3oool20<c<c0D0oooo00@000000<c<c0000000
0000[P3oool003D0oooo00<000000?ooo`3oool0603oool00`3<c<`0oooo0?ooo`040?ooo`80c<c<
1P3oool20<c<c0H0oooo0P3<c<`50?ooo`040000003<c<`0000000000:h0oooo000e0?ooo`030000
003oool0oooo01P0oooo00<0c<c<0?ooo`3oool0103oool20<c<c0H0oooo0P3<c<`60?ooo`80c<c<
1@3oool300000:l0oooo00070?ooo`@000000`3oool2000000@0oooo0P0000030?ooo`050000003o
ool0oooo0?ooo`000000103oool500000080oooo0P00000:0?ooo`8000006@3oool00`3<c<`0oooo
0?ooo`040?ooo`80c<c<1P3oool20<c<c0H0oooo0P3<c<`30?ooo`800000/P3oool000L0oooo00<0
00000?ooo`3oool02@3oool010000000oooo0?ooo`0000030?ooo`030000003oool0000000L0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3o
ool0603oool00`3<c<`0oooo0?ooo`040?ooo`80c<c<1P3oool20<c<c0H0oooo0P3<c<`20?ooo`03
0000003oool0oooo0;80oooo00080?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000
oooo0?ooo`070?ooo`030000003oool0oooo01P0oooo00<0c<c<0?ooo`3oool0103oool20<c<c0H0
oooo0P3<c<`60?ooo`80c<c<0P00002e0?ooo`0000<0oooo0000000000000P0000040?ooo`030000
003oool0oooo00P0oooo0P0000040?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00L0oooo0`00000H0?ooo`030<c<c03oool0oooo00@0
oooo0P3<c<`60?ooo`80c<c<1@3oool400000;H0oooo00070?ooo`040000003oool0oooo000000T0
oooo00<000000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool0000000H0oooo00D000000?oo
o`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0603oool0
0`3<c<`0oooo0?ooo`040?ooo`80c<c<1P3oool20<c<c0D0oooo1000002f0?ooo`00203oool20000
00X0oooo0`00000<0?ooo`800000103oool2000000P0oooo0`00000J0?ooo`030<c<c03oool0oooo
00<0oooo100000050?ooo`80c<c<0`3oool2000000030?ooo`00000000000;L0oooo000c0?ooo`03
0000003oool0000001X0oooo00<0c<c<0?ooo`3oool00`3oool900000080c<c<0`00002l0?ooo`00
;P3oool3000000<0oooo0`00000I0?ooo`030<c<c03oool0oooo00<0oooo0`0000040?ooo`D00000
_P3oool003<0oooo00<000000?ooo`0000006P3oool00`3<c<`0oooo0?ooo`020?ooo`030000003o
ool0oooo00L0oooo0P00002o0?ooo`00<`3oool3000001X0oooo00D0c<c<0?ooo`3oool0oooo0000
003;0?ooo`00=@3oool2000001T0oooo00D0c<c<0?ooo`3oool0oooo000000070?ooo`800000103o
ool2000000<0oooo1000000O0?ooo`800000103oool2000000D0oooo0`00000N0?ooo`800000103o
ool2000000<0oooo0`00000P0?ooo`800000103oool2000000@0oooo0P00000V0?ooo`D000000@3o
ool003D0oooo00<000000?ooo`3oool0603oool0103<c<`0oooo0?ooo`0000070?ooo`040000003o
ool0oooo000000P0oooo00<000000?ooo`3oool07`3oool010000000oooo0?ooo`00000;0?ooo`03
0000003oool0oooo01`0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`00000N
0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000009`3oool00`000000oooo
0?ooo`010?ooo`00=@3oool00`000000oooo0?ooo`0H0?ooo`030<c<c03oool0000000P0oooo00@0
00000?ooo`3oool000002@3oool00`000000oooo0?ooo`0N0?ooo`040000003oool0oooo000000P0
oooo1@00000M0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000007P3oool0
10000000oooo0?ooo`0000080?ooo`040000003oool0oooo000002L0oooo00<000000?ooo`3oool0
0@3oool003D0oooo00<000000?ooo`3oool04P3oool00`3<c<`0oooo0?ooo`020<c<c0040000003<
c<`0oooo000000P0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`0M0?ooo`04
0000003oool0oooo000000P0oooo00@000000?ooo`3oool000007P3oool010000000oooo0?ooo`00
00080?ooo`<000007`3oool010000000oooo0?ooo`0000090?ooo`800000:03oool00`000000oooo
0?ooo`010?ooo`00=@3oool2000000l0oooo0P3<c<`20?ooo`80c<c<00@000000<c<c0000000c<c<
0P0000090?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000007P3oool01000
0000oooo0?ooo`0000090?ooo`030000003oool0000001h0oooo00@000000?ooo`3oool000002@3o
ool00`000000oooo0?ooo`0N0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
00009`3oool00`000000oooo0?ooo`010?ooo`00=@3oool00`000000oooo0?ooo`0<0?ooo`800000
103<c<`00`000000c<c<00000005000000X0oooo0P00000:0?ooo`800000803oool2000000/0oooo
0P00000O0?ooo`8000002P3oool3000001l0oooo0P00000:0?ooo`8000009`3oool2000000<0oooo
000e0?ooo`030000003oool0oooo008000002@3oool00`000000c<c<00000008000000030?ooo`00
000000000<l0oooo000e0?ooo`H00000203oool00`3<c<`000000000000700000=<0oooo000e0?oo
o`030000003oool0000000@000001`3oool00`3<c<`00000000000020?ooo`800000eP3oool003D0
oooo00<000000?ooo`3oool01@0000060?ooo`030<c<c0000000oooo0=X0oooo000e0?ooo`80003o
0P3oool5000000@0oooo00<0c<c<0000003oool0f`3oool003D0oooo0`000?l5000000060<c<c000
0000c<c<0<c<c0000000c<c<fP0000030?ooo`00=@3oool30000o`030<c<c0000000oooo00@00000
00<0c<c<0000003<c<`00P0000080?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0040oooo000e0?ooo`<0003o00@0
00000?ooo`3oool0oooo2000003L0?ooo`00=@3oool30000o`040000003oool0oooo0?ooo`L00000
g@3oool003D0oooo0P000?l00`3oool000000?ooo`030?ooo`800000h@3oool003D0oooo00@0003o
0?ooo`3oool00000i`3oool003D0oooo00@0003o0?ooo`3oool00000i`3oool003D0oooo00@0003o
0000003oool00000i`3oool003D0oooo0P000?l00`3oool000000?ooo`3V0?ooo`00=@3oool30000
o`030000003oool0oooo0>D0oooo000e0?ooo`<0003o00<000000?ooo`3oool0i@3oool003D0oooo
0P000?l00`3oool000000?ooo`3V0?ooo`00=@3oool010000?l000000?ooo`00003W0?ooo`00=@3o
ool010000?l000000?ooo`00003W0?ooo`00=@3oool010000?l000000?ooo`00003W0?ooo`001`3o
ool4000000<0oooo0P0000040?ooo`8000000`3oool01@000000oooo0?ooo`3oool0000000@0oooo
1@0000020?ooo`8000002P3oool20000o`800000i`3oool000L0oooo00<000000?ooo`3oool02@3o
ool010000000oooo0?ooo`0000030?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00L0oooo0`000?l00`000000oooo0?ooo`3U0?ooo`00
203oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
1P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3oool30000o`03
0000003oool0oooo0>D0oooo00090?ooo`030000003oool0oooo00P0oooo0P0000040?ooo`030000
003oool0000000L0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0
oooo0P000?l2000003<0oooo00<0003o0?ooo`3oool01@3oool00`000?l0oooo0?ooo`040?ooo`03
0000o`3oool0oooo0:80oooo00070?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3o
ool00P3oool01@000000oooo0?ooo`3oool0000000H0oooo00D000000?ooo`3oool0oooo00000002
0?ooo`030000003oool0oooo00L0oooo00@0c<c<0000o`3oool00000<P3oool20000o`H0oooo0P00
0?l50?ooo`80003o1@3oool00`000?l0oooo0?ooo`2L0?ooo`00203oool2000000X0oooo0`00000<
0?ooo`800000103oool2000000P0oooo0`0000000`000?l0oooo0000000b0?ooo`80003o1P3oool2
0000o`D0oooo0P000?l50?ooo`030000o`3oool0oooo09`0oooo000c0?ooo`060000003oool00000
0000o`3oool00000<@3oool30000o`H0oooo0P000?l50?ooo`<0003o0`3oool20000oih0oooo000^
0?ooo`<000000`3oool01@000000003o0000o`3oool000000300oooo1@000?l50?ooo`80003o1@3o
ool40000o`80oooo00<0003o0?ooo`000?l0W@3oool003<0oooo00<000000?ooo`000?l00P000?l0
0`000000oooo0?ooo`0^0?ooo`D0003o1@3oool20000o`D0oooo10000?l20?ooo`030000o`3oool0
003o09d0oooo000c0?ooo`8000000`000?l00`000000oooo0?ooo`0_0?ooo`80003o00<0oooo0000
o`3oool0103oool20000o`D0oooo0`000?l30?ooo`030000o`3oool0003o09d0oooo000e0?ooo`80
003o0P00000a0?ooo`040000o`3oool0/k?o0000o`D0oooo0P000?l50?ooo`030000o`3oool0003o
00<0oooo0`000?nE0?ooo`80/k?o1P3oool003D0oooo00@000000000o`0000000000:@3oool20000
o`H0oooo00@0003o0?ooo`3oool0003o1@3oool20000o`080;>co`3oool0oooo0000o`2c/ol0003o
0?ooo`000?l30?ooo`@0003oU03oool20;>co`H0oooo000e0?ooo`80003o0P00000X0?ooo`@0003o
1@3oool010000?l0oooo0?ooo`000?l40?ooo`050;>co`000?l0003o0;>co`3oool00P000?l0102c
/ol0003o0?ooo`000?l30?ooo`@0003oU03oool20;>co`H0oooo000e0?ooo`<0003o00<000000?oo
o`3oool07`3oool20000o`D0oooo0P000?l00`2c/ol0003o0?ooo`040?ooo`040000o`3oool0oooo
0000o`@0oooo00D0/k?o0000o`3oool0003o0?ooo`040000o`030?ooo`000?l0oooo0080oooo0`00
0?nA0?ooo`80/k?o0P3oool20;>co`H0oooo000e0?ooo`<0003o00<000000?ooo`3oool07P3oool4
0000o`@0oooo00@0003o0;>co`2c/ol0003o103oool01@000?l0/k?o0?ooo`3oool0003o00@0oooo
00@0/k?o0000o`000?l0003o0P3oool20000o`80oooo00<0003o0?ooo`3oool00P000?l00`3oool0
003o0?ooo`2@0?ooo`80/k?o0P3oool20;>co`H0oooo000e0?ooo`<0003o00<000000?ooo`3oool0
703oool30000o`030;>co`000?l0003o00@0oooo00D0003o0;>co`2c/ol0oooo0000o`030?ooo`04
0000o`2c/ol0oooo0?ooo`80003o0P3oool00`000?l0/k?o0000o`030000o`80oooo00D0003o0?oo
o`3oool0003o0?ooo`040000o`030;>co`000?l0oooo08l0oooo0P2c/ol20?ooo`80/k?o1P3oool0
03D0oooo0`000?l00`000000oooo0?ooo`0K0?ooo`@0003o0P2c/ol00`000?l0oooo0?ooo`020000
o`050?ooo`2c/ol0/k?o0?ooo`000?l00`3oool00`000?l0/k?o0?ooo`040000o`030?ooo`000?l0
003o00@0003o0P3oool01@000?l0oooo0?ooo`000?l0oooo00@0003o00<0/k?o0000o`3oool00P3o
ool00`000?l0oooo0?ooo`270?ooo`030;>co`3oool0oooo0080/k?o0P3oool20;>co`H0oooo000D
0?ooo`8000000`3oool01@000000oooo0?ooo`3oool0000000@0oooo1@0000020?ooo`8000002P3o
ool20000o`8000006@3oool00`000?l0oooo0000o`050000o`80/k?o1P000?l20;>co`030?ooo`00
0?l0oooo0080oooo00<0003o0;>co`3oool010000?l20?ooo`@0003o0`3oool01P000?l0oooo0?oo
o`2c/ol0003o0?ooo`80003o00<0oooo0;>co`000?l02@000?lI0?ooo`80003o0P3oool20000o`80
oooo0P000?l00`3oool0003o0000o`020?ooo`80003o0P3oool20000o`80oooo0P000?mA0?ooo`80
/k?o0P3oool20;>co`80oooo0P2c/ol60?ooo`004`3oool010000000oooo0?ooo`0000030?ooo`03
0000003oool0000000L0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00L0oooo1`000?l30?ooo`<0003o3`3oool70000o`@0/k?o1@000?l20;>co`040?ooo`2c/ol0003o
0000o`80oooo00@0003o0;>co`3oool0oooo0P000?l30?ooo`040000o`3oool0oooo0000o`80oooo
00/0/k?o0000o`3oool0oooo0;>co`000?l0oooo0000o`2c/ol0oooo0;>co`040000o`030;>co`00
0?l0003o00L0003o00<0oooo0000o`000?l00P3oool20000o`80oooo0P000?l20?ooo`80003o0P3o
ool20000o`030?ooo`000?l0003o01T0003o00<0oooo0000o`000?l00P3oool20000o`80oooo0P00
0?l00`3oool0003o0000o`120?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol60?ooo`005P3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo0000
00020?ooo`030000003oool0oooo00L0oooo1P000?l00`2c/ol0003o0000o`0=0000o`@0oooo1P00
0?l0102c/ol0003o0;>co`3oool30;>co`030000o`2c/ol0003o0080003o00@0/k?o0?ooo`3oool0
/k?o0`000?l00`3oool0003o0;>co`030?ooo`050000o`3oool0oooo0?ooo`000?l00P3oool01000
0?l0oooo0?ooo`000?l30?ooo`040;>co`000?l0oooo0000o`80oooo00D0/k?o0000o`000?l0003o
0;>co`180000o`030?ooo`000?l0003o0080oooo0P000?lg0?ooo`030;>co`3oool0oooo0080/k?o
0P3oool20;>co`80oooo0P2c/ol60?ooo`00503oool2000000@0oooo00<000000?ooo`0000001`3o
ool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3ooolC0000o`030;>c
o`000?l0003o00<0oooo10000?l20;>co`050000o`2c/ol0/k?o0;>co`3oool00P2c/ol00`3oool0
/k?o0;>co`020000o`050;>co`3oool0oooo0?ooo`2c/ol010000?l00`2c/ol0oooo0?ooo`020?oo
o`050000o`3oool0oooo0?ooo`000?l00P3oool010000?l0oooo0;>co`000?l30?ooo`040;>co`00
0?l0oooo0000o`80oooo0P2c/ol20000o`040;>co`000?l0/k?o0000o`H0/k?o0`000?l01@2c/ol0
003o0000o`000?l0/k?o00<0003o00D0/k?o0000o`000?l0003o0;>co`030000o`050;>co`000?l0
003o0000o`2c/ol01P000?l01P2c/ol0003o0000o`000?l0/k?o0?ooo`80003o00D0/k?o0?ooo`00
0?l0003o0;>co`030000o`050;>co`000?l0003o0000o`2c/ol050000?l00`3oool0003o0000o`02
0?ooo`80003o;P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`H0oooo000D0?oo
o`030000003oool0oooo0080oooo00D000000?ooo`3oool0oooo000000060?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`070?ooo`L0003o0P2c/ol50000o`060;>co`00
0?l0/k?o0;>co`000?l0/k?o1`000?l01@2c/ol0003o0;>co`2c/ol0oooo00<0/k?o103oool30;>c
o`030?ooo`2c/ol0oooo00<0oooo00D0/k?o0000o`000?l0003o0;>co`040?ooo`050000o`3oool0
oooo0?ooo`000?l00P3oool010000?l0oooo0;>co`000?l40?ooo`<0003o0P3oool20;>co`80003o
0P3oool20;>co`80oooo1@2c/ol00`3oool0003o0;>co`020?ooo`060000o`2c/ol0oooo0?ooo`00
0?l0/k?o0P3oool010000?l0/k?o0?ooo`3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool2
0;>co`80oooo00D0/k?o0?ooo`3oool0oooo0;>co`030?ooo`050;>co`3oool0oooo0?ooo`2c/ol0
0P3oool20;>co`80oooo0P2c/ol0103oool0003o0;>co`2c/ol30000o`030?ooo`000?l0003o00h0
003o00<0oooo0000o`000?l00P3oool20000ob<0oooo00<0/k?o0?ooo`3oool00P2c/ol20?ooo`80
/k?o0P3oool20;>co`80oooo0P2c/ol60?ooo`00503oool3000000`0oooo0P0000040?ooo`800000
2P3oool:0000o`<0oooo00<0003o0;>co`3oool00P2c/ol010000?l0/k?o0000o`2c/ol20000o`05
0;>co`000?l0003o0;>co`3oool00P2c/ol0103oool0/k?o0;>co`2c/ol40?ooo`<0/k?o00<0oooo
0;>co`3oool00`3oool01@2c/ol0003o0000o`000?l0/k?o00@0oooo00D0003o0?ooo`3oool0oooo
0000o`020?ooo`040;>co`000?l0/k?o0000o`<0oooo00@0003o0;>co`000?l0003o0`3oool00`2c
/ol0oooo0?ooo`020?ooo`80/k?o1P3oool0102c/ol0oooo0?ooo`2c/ol30?ooo`050;>co`3oool0
oooo0?ooo`2c/ol00P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c
/ol20?ooo`050;>co`3oool0oooo0?ooo`2c/ol00`3oool00`2c/ol0oooo0?ooo`020;>co`80oooo
0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol0103oool0003o0;>co`3oool20000o`05
0;>co`3oool0003o0000o`2c/ol03`000?lP0?ooo`070;>co`3oool0oooo0;>co`3oool0/k?o0?oo
o`020;>co`80oooo0P2c/ol20?ooo`80/k?o1P3oool003<0oooo0P0000050000o`T0oooo00<0/k?o
0?ooo`2c/ol00P2c/ol01@3oool0/k?o0;>co`2c/ol0003o0080/k?o00<0003o0;>co`2c/ol00`2c
/ol00`3oool0/k?o0;>co`040?ooo`040;>co`3oool0/k?o0;>co`D0oooo00<0/k?o0000o`000?l0
0P000?l40?ooo`050000o`3oool0oooo0?ooo`000?l00P3oool0102c/ol0003o0000o`000?l30?oo
o`040000o`2c/ol0003o0000o`<0oooo00<0/k?o0?ooo`3oool00P3oool20;>co`H0oooo00@0/k?o
0?ooo`3oool0/k?o0`3oool01@2c/ol0oooo0?ooo`3oool0/k?o0080oooo0P2c/ol20?ooo`80/k?o
0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool01@2c/ol0oooo0?ooo`3oool0/k?o00<0
oooo00<0/k?o0?ooo`3oool00P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80
/k?o0P3oool01@2c/ol0oooo0?ooo`3oool0/k?o0080oooo0P2c/ol0103oool0003o0000o`2c/ol:
0000o`030;>co`000?l0oooo0080003o703oool01`2c/ol0oooo0?ooo`2c/ol0oooo0;>co`3oool0
0P2c/ol20?ooo`060;>co`3oool0/k?o0;>co`3oool0/k?o1P3oool003<0oooo00<000000?ooo`00
0?l00P000?l00`2c/ol0003o0?ooo`080?ooo`030;>co`3oool0/k?o0080/k?o00D0oooo0;>co`2c
/ol0/k?o0?ooo`030;>co`030?ooo`2c/ol0/k?o0080/k?o00<0oooo0;>co`2c/ol01P3oool20;>c
o`H0oooo0`000?l50?ooo`050000o`3oool0oooo0?ooo`000?l00P3oool0102c/ol0003o0000o`00
0?l40?ooo`<0003o0`3oool00`2c/ol0oooo0?ooo`020?ooo`80/k?o1P3oool0102c/ol0oooo0?oo
o`2c/ol30?ooo`050;>co`3oool0oooo0?ooo`2c/ol00P3oool20;>co`80oooo0P2c/ol20?ooo`80
/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`050;>co`3oool0oooo0?ooo`2c/ol00`3oool00`2c
/ol0oooo0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?oo
o`050;>co`3oool0oooo0?ooo`2c/ol00P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o2P000?l0
0`2c/ol0003o0?ooo`020000oa@0oooo00<0/k?o0?ooo`3oool00P2c/ol20?ooo`0:0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`80/k?o00<0oooo0;>co`3oool01@3oool002h0
oooo0`0000030?ooo`03000000000?l0003o0080003o2`3oool01@2c/ol0oooo0;>co`2c/ol0oooo
00<0/k?o00D0oooo0;>co`2c/ol0/k?o0?ooo`020;>co`030?ooo`2c/ol0oooo0080/k?o1P3oool2
0;>co`H0oooo00<0/k?o0000o`2c/ol01@3oool01@000?l0oooo0?ooo`3oool0003o0080oooo00<0
/k?o0000o`000?l01P3oool20000oa00oooo00<0/k?o0?ooo`3oool00P2c/ol20?ooo`80/k?o0P3o
ool00`2c/ol0oooo0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c
/ol20?ooo`030;>co`3oool0oooo0080/k?o0`3oool00`2c/ol0oooo0?ooo`020;>co`80oooo0P2c
/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`050;>co`3oool0oooo0?ooo`2c/ol0
0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`<0003o00<0/k?o0000o`000?l0
1P000?l00`3oool0003o0000o`0@0?ooo`0=0;>co`3oool0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol00`3oool0/k?o0?ooo`050?ooo`00<`3o
ool00`000000oooo0000o`030000o`/0oooo00D0/k?o0?ooo`2c/ol0/k?o0?ooo`040;>co`030?oo
o`2c/ol0/k?o00@0oooo00@0/k?o0?ooo`3oool0/k?o1P3oool20;>co`H0oooo00<0/k?o0?ooo`2c
/ol01@3oool01@000?l0oooo0?ooo`3oool0003o0080oooo00<0/k?o0000o`000?l01P3oool20000
oa00oooo00<0/k?o0?ooo`3oool00P2c/ol20?ooo`80/k?o0P3oool00`2c/ol0oooo0?ooo`020;>c
o`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o00@0oooo0;>c
o`3oool0/k?o0P3oool00`2c/ol0oooo0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool2
0;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`030?ooo`2c/ol0/k?o0080oooo0P2c/ol2
0?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20000o`80/k?o1P000?l00`2c/ol0003o0?ooo`02
0000o`/0oooo0P2c/ol20?ooo`0:0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`80/k?o0P3oool20;>co`030?ooo`2c/ol0oooo00D0oooo000c0?ooo`80000010000?lB0?ooo`80
/k?o00<0oooo0;>co`2c/ol03`3oool00`2c/ol0oooo0?ooo`040?ooo`80/k?o1P3oool01@2c/ol0
003o0?ooo`000?l0/k?o0080oooo00<0/k?o0000o`000?l01P3oool20000oa00oooo00<0/k?o0?oo
o`3oool00P2c/ol20?ooo`80/k?o0P3oool00`2c/ol0oooo0?ooo`020;>co`80oooo0P2c/ol20?oo
o`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o00@0oooo0;>co`3oool0/k?o0P3oool0
0`2c/ol0oooo0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol2
0?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3o
ool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P000?l00`2c/ol0003o0000o`050000o`030;>co`00
0?l0oooo00D0oooo0P2c/ol20?ooo`0>0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o
0?ooo`2c/ol0oooo0;>co`3oool20;>co`80oooo0P2c/ol00`3oool0/k?o0?ooo`050?ooo`00=@3o
ool010000000003o0000o`00000C0?ooo`030;>co`3oool0oooo01L0oooo0P2c/ol60?ooo`040;>c
o`000?l0oooo0000o`<0oooo00<0/k?o0?ooo`2c/ol01P3oool20000oa00oooo00<0/k?o0?ooo`3o
ool00P2c/ol20?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?oo
o`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`02
0;>co`80oooo00@0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c
/ol20?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o
0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80003o00<0/k?o0000o`000?l0
0`000?l0103oool0003o0;>co`2c/ol20?ooo`0B0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o0P3o
ool00`2c/ol0oooo0?ooo`030?ooo`00=@3oool010000000c<c<00000000000C0?ooo`030;>co`3o
ool0oooo01L0oooo0P2c/ol60?ooo`040;>co`000?l0oooo0000o`<0oooo00<0/k?o0?ooo`2c/ol0
1P3oool20;>coa00oooo00<0/k?o0?ooo`3oool00P2c/ol20?ooo`80/k?o00D0oooo0;>co`3oool0
/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80
/k?o00d0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080
/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o00T0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3o
ool20;>co`080?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol20000o`030;>co`000?l0003o
0080003o01D0/k?o0?ooo`000?l0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool00`2c/ol0
oooo0?ooo`030?ooo`00=@3oool010000000c<c<00000000000]0?ooo`80/k?o1P3oool0102c/ol0
003o0?ooo`000?l30?ooo`030;>co`3oool0/k?o00H0oooo0P2c/ol@0?ooo`030;>co`3oool0oooo
0080/k?o0P3oool20;>co`050?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool2
0;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`0=0?ooo`2c/ol0oooo0;>co`3oool0/k?o
0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool2
0;>co`090?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80
oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`0?0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo00T0003o0100/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o0P3o
ool00`2c/ol0oooo0?ooo`030?ooo`00=@3oool010000000c<c<00000000000]0?ooo`80/k?o1P3o
ool0102c/ol0003o0?ooo`000?l30?ooo`80/k?o1`3oool20;>coa00oooo00<0/k?o0?ooo`3oool0
0P2c/ol20?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80
/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o00d0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80
/k?o00d0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080
/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0180oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol60000o`040?ooo`2c
/ol0003o0?ooo`80/k?o0P3oool20;>co`80oooo00@0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80
/k?o0P3oool00`2c/ol0oooo0?ooo`030?ooo`00=@3oool010000000c<c<00000000000]0?ooo`80
/k?o1P3oool0102c/ol0003o0000o`000?l30?ooo`80/k?o6@3oool01@2c/ol0oooo0;>co`3oool0
/k?o0080oooo0P2c/ol01@3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c
/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol03@3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c
/ol03@3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c
/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol04P3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`80oooo2@000?l00`2c
/ol0oooo0?ooo`020;>co`80oooo00@0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o0P3oool0
0`2c/ol0oooo0?ooo`030?ooo`00=@3oool010000000c<c<00000000000^0?ooo`030;>co`3oool0
oooo00@0oooo00@0/k?o0000o`000?l0003o0`3oool20;>coaT0oooo00D0/k?o0?ooo`2c/ol0oooo
0;>co`020?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80
/k?o0P3oool20;>co`040?ooo`2c/ol0oooo0;>co`80oooo0P2c/ol03@3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool20;>c
o`0E0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c/ol03P3oool0
/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0P3oool00`2c
/ol0oooo0?ooo`040?ooo`H0003o00<0/k?o0000o`3oool00P2c/ol20?ooo`80/k?o0P3oool20;>c
o`80oooo0P2c/ol20?ooo`030;>co`3oool0oooo00<0oooo000e0?ooo`040000003<c<`000000000
03D0oooo00@0/k?o0000o`000?l0003o0`3oool20;>coaT0oooo00h0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`80/k?o0P3oool0102c/ol0oooo0;>c
o`3oool20;>co`80oooo0P2c/ol0103oool0/k?o0?ooo`2c/ol20?ooo`80/k?o00d0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80
oooo0P2c/ol05@3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol03`3oool0/k?o
0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co``0
oooo1P000?l00`3oool0003o0;>co`020?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`03
0;>co`3oool0oooo00<0oooo00090?ooo`030000003oool0oooo0080oooo0P0000040?ooo`800000
0`3oool01@000000oooo0?ooo`3oool0000000@0oooo1@0000020?ooo`8000002P3oool010000000
c<c<00000000000e0?ooo`030;>co`000?l0/k?o00D0oooo00<0/k?o0?ooo`3oool05`3oool03P2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0P2c/ol2
0?ooo`040;>co`3oool0/k?o0?ooo`80/k?o0P3oool20;>co`040?ooo`2c/ol0oooo0;>co`80oooo
0P2c/ol03@3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0
0P2c/ol20?ooo`80/k?o01l0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o00`0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol20?ooo`030;>co`3oool0oooo00`0oooo20000?l00`3oool0/k?o0;>c
o`020?ooo`80/k?o0P3oool20;>co`80oooo00<0/k?o0?ooo`3oool00`3oool000T0oooo00<00000
0?ooo`3oool01`3oool010000000oooo0?ooo`0000030?ooo`030000003oool0000000L0oooo00D0
00000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00L0oooo00@000000<c<c0000000
0000=@3oool00`2c/ol0oooo0;>co`050?ooo`030;>co`3oool0oooo01L0oooo01H0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o00@0oooo0;>co`3oool0/k?o0P3oool20;>c
o`0=0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>c
o`80oooo0P2c/ol07`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool00P2c/ol0203oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0P3o
ool00`2c/ol0oooo0?ooo`0D0?ooo`@0003o00<0oooo0000o`2c/ol00P3oool20;>co`80oooo0P2c
/ol20?ooo`030;>co`3oool0oooo00<0oooo00090?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`0000000P3o
ool00`000000oooo0?ooo`070?ooo`040000003<c<`00000000003D0oooo0P2c/ol60?ooo`030;>c
o`3oool0oooo01L0oooo01H0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o
01D0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o01l0oooo0;>co`3oool0/k?o
0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o00@0oooo
0;>co`3oool0/k?o0P3oool20;>co`80oooo00<0/k?o0?ooo`3oool05@3oool70000o`030?ooo`2c
/ol0/k?o0080oooo0P2c/ol20?ooo`030;>co`3oool0oooo00<0oooo000:0?ooo`030000003oool0
oooo00L0oooo0P0000040?ooo`030000003oool0000000L0oooo00D000000?ooo`3oool0oooo0000
00020?ooo`030000003oool0oooo00L0oooo00@000000<c<c00000000000=@3oool20;>co`H0oooo
00<0/k?o0?ooo`3oool05`3oool05P2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool20;>co`80oooo
0P2c/ol05@3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol06`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80
/k?o0P3oool20;>coad0oooo10000?l00`3oool0003o0;>co`020?ooo`80/k?o0P3oool00`2c/ol0
oooo0?ooo`030?ooo`001`3oool010000000oooo0?ooo`0000090?ooo`030000003oool0oooo0080
oooo00D000000?ooo`3oool0oooo000000060?ooo`050000003oool0oooo0?ooo`0000000P3oool0
0`000000oooo0?ooo`070?ooo`040000003<c<`00000000003D0oooo0P2c/olP0?ooo`0F0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`80/k?o0P3oool20;>co`0I0?ooo`2c/ol0oooo0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`020;>co`0K0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o7P3oool70000
o`030?ooo`2c/ol0/k?o0080oooo00<0/k?o0?ooo`3oool00`3oool000L0oooo100000090?ooo`<0
0000303oool2000000@0oooo0P0000080?ooo`<0000000<0c<c<000000000000=@3oool20;>cob00
oooo01H0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o01T0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o01/0oooo0;>co`3oool0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool01@2c/ol0oooo0?oo
o`3oool0/k?o0280oooo10000?l00`3oool0/k?o0;>co`020?ooo`030;>co`000?l0003o00<0oooo
000c0?ooo`040000003oool000000<c<c0800000=P3oool00`2c/ol0oooo0?ooo`0N0?ooo`0F0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool0/k?o0?ooo`80/k?o0P3oool20;>co`0K0?ooo`2c/ol0oooo0;>co`3o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o01D0oooo0;>co`3oool0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3o
ool00P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo00<0/k?o0?ooo`3oool09@3oool40000o`05
0;>co`000?l0003o0?ooo`2c/ol00P000?l30?ooo`00;P3oool3000000<0oooo1@00000f0?ooo`03
0;>co`3oool0oooo01h0oooo01X0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o
0?ooo`80/k?o01/0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0
0P2c/ol04@3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool00`2c
/ol0oooo0?ooo`0X0?ooo`@0003o00@0oooo0;>co`000?l0003o0`3oool003<0oooo00@000000?oo
o`000000oooo0P00001G0?ooo`0J0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool20;>co`0K0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o
0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0080/k?o0140oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>cobh0oooo1000
0?l00`2c/ol0003o0000o`030?ooo`00<`3oool3000000030?ooo`000000000005L0oooo01P0/k?o
0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool20;>co`0I0?ooo`2c/ol0oooo0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol04@3oool0/k?o0?ooo`2c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>c
o`80oooo0P2c/ol20?ooo`030;>co`3oool0oooo02l0oooo1@000?l30?ooo`00=@3oool4000005L0
oooo01P0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool20;>co`0E0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o
0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c/ol02@3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c
/ole0?ooo`<0003o103oool003D0oooo00@000000?ooo`0000000000E`3oool0602c/ol0oooo0;>c
o`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?oo
o`2c/ol0oooo0;>co`3oool0/k?o0?ooo`80/k?o01D0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c
/ol20?ooo`80/k?o0P3oool20;>co`090?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0080/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>coc`0oooo000e0?ooo`04
0000003oool00000000005L0oooo01P0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool2
0;>co`0E0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c
/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c/ol0203o
ool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0P3oool00`2c/ol0oooo0?ooo`020;>co`80oooo
0P2c/ol20?ooo`80/k?o@03oool003D0oooo1000001G0?ooo`0H0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0P2c/ol20?ooo`80/k?o0140oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?oo
o`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`050;>co`3oool0oooo0?ooo`2c/ol00P3oool2
0;>co`<0oooo00<0/k?o0?ooo`3oool0?P3oool003D0oooo1000001G0?ooo`0D0;>co`3oool0/k?o
0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo
0;>co`3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0140oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol2
0?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`050;>co`3oool0oooo0?ooo`2c/ol00`3o
ool00`2c/ol0oooo0?ooo`120?ooo`00=@3oool4000005L0oooo00<0/k?o0?ooo`2c/ol00P3oool2
0;>co`090?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80
oooo0P2c/ol20?ooo`80/k?o0140oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o
0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool2
0;>co`80oooo0P2c/ol20?ooo`050;>co`3oool0oooo0?ooo`2c/ol0B03oool003D0oooo1000001H
0?ooo`80/k?o0P3oool20;>co`090?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0080
/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0140oooo0;>co`3oool0/k?o0?ooo`2c/ol0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol2
0?ooo`80/k?o0P3oool20;>co`80oooo00D0/k?o0?ooo`3oool0oooo0;>co`030?ooo`030;>co`3o
ool0oooo04H0oooo000e0?ooo`040000003oool00000000005P0oooo0P2c/ol20?ooo`80/k?o00T0
oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool20;>c
o`80oooo0P2c/ol20?ooo`80/k?o00d0oooo0;>co`3oool0/k?o0?ooo`2c/ol0oooo0;>co`3oool0
/k?o0?ooo`2c/ol0oooo0080/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool01@2c
/ol0oooo0?ooo`3oool0/k?o04`0oooo000C0?ooo`D0000000<0oooo0000003oool00P3oool00`00
0000oooo0?ooo`020?ooo`D000000P3oool2000000X0oooo0`00001I0?ooo`80/k?o0P3oool20;>c
o`050?ooo`2c/ol0oooo0;>co`3oool00P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol2
0?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o00<0oooo0;>co`2c/ol00P3oool2
0;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo00<0/k?o0?ooo`3oool0CP3oool0
01D0oooo00<000000?ooo`3oool00P3oool00`000000oooo000000070?ooo`050000003oool0oooo
0?ooo`0000000P3oool00`000000oooo0?ooo`070?ooo`030000003oool0000005T0oooo0P2c/ol2
0?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80/k?o0P3o
ool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo00<0/k?o0?ooo`3oool00P2c
/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`050;>co`3oool0oooo0?ooo`2c/ol0
D03oool001D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`060?ooo`050000003o
ool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`070?ooo`030000003oool0000005T0oooo
0P2c/ol20?ooo`80/k?o00D0oooo0;>co`3oool0/k?o0?ooo`020;>co`80oooo0P2c/ol20?ooo`80
/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo00<0/k?o0?ooo`3o
ool00P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo0P2c/ol20?ooo`030;>co`3oool0oooo0580
oooo000E0?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000001`3oool01@000000oooo
0?ooo`3oool000000080oooo00<000000?ooo`3oool01`3oool3000005T0oooo0P2c/ol20?ooo`80
/k?o0P3oool20;>co`030?ooo`2c/ol0/k?o0080oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80
oooo0P2c/ol20?ooo`80/k?o0P3oool20;>co`80oooo00<0/k?o0?ooo`3oool00P2c/ol20?ooo`80
/k?o0P3oool20;>co`<0oooo00<0/k?o0?ooo`3oool0E@3oool001D0oooo00D000000?ooo`3oool0
oooo000000030?ooo`030000003oool0oooo00@0oooo00D000000?ooo`3oool0oooo000000020?oo
o`030000003oool0oooo0?80oooo000D0?ooo`8000003@3oool2000000@0oooo0P0000090?ooo`03
0000003oool0oooo0>T0oooo000d0?ooo`030000003oool0oooo0>T0oooo000^0?ooo`<00000103o
ool00`000000oooo0?ooo`3X0?ooo`00=@3oool00`000000oooo0?ooo`3X0?ooo`00<`3oool30000
0>X0oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?oo
ob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?oo
ob40oooo003o0?ooob40oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.234127, -1.24409*^-07, \
0.00434554, 1.40625*^-09}}]
}, Open  ]],

Cell["\<\
You can see from the plot that the error is somewhat smaller when the \
quantity is computed algebraically along with the solution.\
\>", "Text"],

Cell[TextData[{
  "The remainder of this documentation will focus on suboptions of the two \
possible settings for the ",
  StyleBox["ImplicitSolver", "MR"],
  " option, which can be ",
  StyleBox["Newton", "MR"],
  " or ",
  StyleBox["GMRES", "MR"],
  ". With ",
  StyleBox["Newton", "MR"],
  ", the Jacobian or an approximation to it is computed and the linear system \
is solved to find the Newton step. On the other hand, ",
  StyleBox["GMRES", "MR"],
  " is an iterative method and, rather than computing the entire Jacobian, a \
directional derivative is computed for each iterative step."
}], "Text"],

Cell[TextData[{
  "The ",
  StyleBox["Newton", "MR"],
  " method has one method option, ",
  StyleBox["LinearSolveMethod", "MR"],
  ", which you can use to tell ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " how to solve the linear equations required to find the Newton step. There \
are several possible values for this option."
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {
            StyleBox["Automatic",
              "MR"], Cell[TextData[{
              "this is the default. Automatically switch between using the ",
              StyleBox["Mathematica",
                FontSlant->"Italic"],
              " ",
              StyleBox["LinearSolve", "MR"],
              " and ",
              StyleBox["Band ", "MR"],
              
              "methods depending on the band width of the Jacobian. For \
systems with larger band width, this will automatically switch to a direct \
sparse solver for large systems with sparse Jacobians "
            }], "Text"]},
          {Cell[TextData[StyleBox["Band", "MR"]]], Cell["\<\
use the IDA band method (see the IDA user manual for more information)\
\>", "Text"]},
          {Cell[TextData[StyleBox["Dense", "MR"]]], Cell["\<\
use the IDA dense method (see the IDA user manual for more information) \
\>", "Text"]}
          }], TextForm]], "DefinitionBox"],

Cell[TextData[{
  "Possible settings for the ",
  StyleBox["LinearSolveMethod", "MR"],
  " option. "
}], "Caption"],

Cell[TextData[{
  "The ",
  StyleBox["GMRES", "MR"],
  " method may be substantially faster, but is typically quite a bit more \
tricky to use because to really be effective typically requires a \
preconditioner, which can be supplied via a method option. There are also \
some other method options that control the Krylov subspace process. To use \
these, refer to the IDA user guide [",
  ButtonBox["HT99",
    ButtonData:>{"References.nb", "HT99"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["GMRES method option name", "SO10"]]], Cell[
            TextData[StyleBox["default value", "SO10"]]], " "},
          {Cell[TextData[StyleBox["Preconditioner", "MR"]]], 
            StyleBox["Automatic",
              "MR"], Cell[TextData[{
              "a ",
              StyleBox["Mathematica",
                FontSlant->"Italic"],
              " function that returns another function that preconditions"
            }], "Text"]},
          {Cell[TextData[StyleBox["OrthogonalizationType", "MR"]]], Cell[
            TextData[StyleBox["ModifiedGramSchmidt", "MR"]]], Cell[TextData[{
              "this can also be ",
              StyleBox["ClassicalGramSchmidt (s", "MR"],
              "ee variable ",
              StyleBox["gstype", "MR"],
              " in the IDA User guide)"
            }], "Text"]},
          {Cell[TextData[StyleBox["MaxKrylovSubspaceDimension", "MR"]]], 
            StyleBox["Automatic",
              "MR"], Cell[TextData[{
              "maximum susbspace dimension (see variable ",
              StyleBox["maxl", "MR"],
              " in the IDA user guide)"
            }], "Text"]},
          {Cell[TextData[StyleBox["MaxKrylovRestarts", "MR"]]], 
            StyleBox["Automatic",
              "MR"], Cell[TextData[{
              "maximum number of restarts (see variable ",
              StyleBox["maxrs", "MR"],
              " in the IDA user guide)"
            }]]}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  StyleBox["GMRES", "MR"],
  " method options. "
}], "Caption"],

Cell["\<\
As an example problem, consider a two-dimensional Burgers\[CloseCurlyQuote] \
equation. \
\>", "Text"],

Cell[BoxData[
    \(TraditionalForm\`u\_t = \[Nu]\ \((u\_xx + 
              u\_yy)\) - \(1\/2\) \((\((u\^2)\)\_x + \((u\^2)\)\_y)\)\)], \
"DisplayMath"],

Cell[TextData[{
  "This can typically be solved with an ordinary differential equation \
solver, but in this example two things are achieved by using the DAE solver. \
First, boundary conditions are enforced as algebraic conditions. Second, ",
  StyleBox["NDSolve", "MR"],
  " is forced to use conservative differencing by using an algebraic term. \
For comparison, a known exact solution will be used for initial and boundary \
conditions."
}], "Text"],

Cell["\<\
This defines a function that satisfies Burger\[CloseCurlyQuote]s equation.\
\>", "MathCaption"],

Cell[BoxData[
    \(Bsol[t_, \ x_, \ y_]\  = \ 
      1/\((1 + Exp[\((x + y - t)\)/\((2\ \[Nu])\)])\); \)], "Input",
  CellLabel->"In[48]:="],

Cell["\<\
This defines initial and boundary conditions for the unit square consistent \
with the exact solution.\
\>", "MathCaption"],

Cell[BoxData[{
    \(ic\  = \ 
      u[0, x, y]\  \[Equal] \ Bsol[0, x, y]; \), "\[IndentingNewLine]", 
    \(bc\  = \ {\[IndentingNewLine]u[t, 0, y]\  \[Equal] \ Bsol[t, 0, y], \ 
        u[t, 1, y]\  \[Equal] \ 
          Bsol[t, 1, y], \[IndentingNewLine]u[t, x, 0]\  \[Equal] \ 
          Bsol[t, x, 0], \ 
        u[t, x, 1]\  \[Equal] \ Bsol[t, x, 1]}; \)}], "Input",
  CellLabel->"In[49]:="],

Cell["This defines the differential equation.", "MathCaption"],

Cell[BoxData[
    \(de\  = \ 
      D[u[t, x, y], 
          t]\  \[Equal] \ \[Nu] \((\ 
              D[u[t, x, y], x, x]\  + D[u[t, x, y], y, y])\)\  - 
          u[t, x, y] \((D[u[t, x, y], x]\  + \ 
                D[u[t, x, y], y])\); \)], "Input",
  CellLabel->"In[51]:="],

Cell[TextData[{
  "This sets the diffusion constant \[Nu] to a value for which we can find a \
solution in a reasonable amount of time and shows a plot of the solution at \
",
  StyleBox["t",
    FontSlant->"Italic"],
  " == 1."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\[Nu]\  = \ 0.025; \), "\[IndentingNewLine]", 
    \(Plot3D[Bsol[1, x, y], {x, 0, 1}, {y, 0, 1}]; \)}], "Input",
  CellLabel->"In[52]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-1.55041e-017 1.04977 -0.0679587 1.04977 [
[.02757 .24618 -6 -8.70276 ]
[.02757 .24618 0 .29724 ]
[.14598 .19958 -17.886 -9 ]
[.14598 .19958 .11404 0 ]
[.27013 .15073 -17.1354 -9 ]
[.27013 .15073 .86461 0 ]
[.40046 .09948 -16.3848 -9 ]
[.40046 .09948 1.61518 0 ]
[.53743 .04564 -15.6343 -9 ]
[.53743 .04564 2.36575 0 ]
[.68156 -0.01098 -4.96123 -9 ]
[.68156 -0.01098 1.03877 0 ]
[.70096 -0.00478 0 -6.26206 ]
[.70096 -0.00478 6 2.73794 ]
[.76745 .09703 0 -6.16187 ]
[.76745 .09703 18 2.83813 ]
[.82812 .18993 0 -6.07246 ]
[.82812 .18993 18 2.92754 ]
[.88369 .27502 0 -5.99218 ]
[.88369 .27502 18 3.00782 ]
[.93478 .35327 0 -5.9197 ]
[.93478 .35327 18 3.0803 ]
[.98191 .42546 0 -5.85393 ]
[.98191 .42546 6 3.14607 ]
[.02411 .26511 -6 -2.74232 ]
[.02411 .26511 0 6.25768 ]
[.01568 .31795 -24 -2.78699 ]
[.01568 .31795 0 6.21301 ]
[.00696 .37264 -18 -2.83339 ]
[.00696 .37264 0 6.16661 ]
[-0.00208 .42927 -24 -2.88161 ]
[-0.00208 .42927 0 6.11839 ]
[-0.01144 .48795 -6 -2.93176 ]
[-0.01144 .48795 0 6.06824 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.1552 .20892 m
.15981 .21359 L
s
[(0.2)] .14598 .19958 .98733 1 Mshowa
.27893 .16047 m
.28333 .16533 L
s
[(0.4)] .27013 .15073 .90393 1 Mshowa
.40878 .10962 m
.41294 .11469 L
s
[(0.6)] .40046 .09948 .82054 1 Mshowa
.54521 .0562 m
.54911 .06148 L
s
[(0.8)] .53743 .04564 .73714 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(1)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.18558 .19702 m
.18832 .19985 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.3108 .14799 m
.31341 .15094 L
s
.34306 .13535 m
.34564 .13834 L
s
.37572 .12257 m
.37826 .12558 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.51046 .06981 m
.51284 .07294 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.68874 0 m
.68263 .00239 L
s
[(0)] .70096 -0.00478 -1 .39157 Mshowa
.75514 .10158 m
.74899 .10385 L
s
[(0.2)] .76745 .09703 -1 .3693 Mshowa
.81573 .19425 m
.80953 .19642 L
s
[(0.4)] .82812 .18993 -1 .34944 Mshowa
.87123 .27915 m
.865 .28122 L
s
[(0.6)] .88369 .27502 -1 .3316 Mshowa
.92227 .35722 m
.91601 .35919 L
s
[(0.8)] .93478 .35327 -1 .31549 Mshowa
.96935 .42924 m
.96306 .43113 L
s
[(1)] .98191 .42546 -1 .30087 Mshowa
.125 Mabswid
.70593 .0263 m
.70226 .02771 L
s
.72272 .05198 m
.71904 .05338 L
s
.73912 .07706 m
.73543 .07845 L
s
.7708 .12553 m
.7671 .12688 L
s
.78611 .14895 m
.78241 .15028 L
s
.80108 .17185 m
.79737 .17317 L
s
.83006 .21617 m
.82633 .21745 L
s
.84407 .23761 m
.84035 .23888 L
s
.8578 .2586 m
.85407 .25986 L
s
.88439 .29928 m
.88065 .3005 L
s
.89727 .31899 m
.89353 .3202 L
s
.9099 .3383 m
.90615 .3395 L
s
.93439 .37576 m
.93063 .37693 L
s
.94627 .39394 m
.94251 .3951 L
s
.95792 .41176 m
.95416 .41291 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.03634 .26033 m
.04245 .25795 L
s
[(0)] .02411 .26511 1 -0.3906 Mshowa
.02795 .31328 m
.03408 .31095 L
s
[(0.25)] .01568 .31795 1 -0.38067 Mshowa
.01926 .36808 m
.02541 .3658 L
s
[(0.5)] .00696 .37264 1 -0.37036 Mshowa
.01027 .42483 m
.01644 .42261 L
s
[(0.75)] -0.00208 .42927 1 -0.35964 Mshowa
.00095 .48363 m
.00715 .48147 L
s
[(1)] -0.01144 .48795 1 -0.3485 Mshowa
.125 Mabswid
.03468 .27078 m
.03835 .26935 L
s
.03301 .2813 m
.03669 .27988 L
s
.03134 .29189 m
.03501 .29047 L
s
.02965 .30255 m
.03332 .30114 L
s
.02623 .32409 m
.02991 .3227 L
s
.02451 .33497 m
.02819 .33359 L
s
.02277 .34593 m
.02646 .34455 L
s
.02102 .35697 m
.02471 .35559 L
s
.01749 .37927 m
.02118 .37791 L
s
.0157 .39054 m
.0194 .38919 L
s
.0139 .40189 m
.0176 .40054 L
s
.01209 .41332 m
.01579 .41198 L
s
.00843 .43642 m
.01214 .4351 L
s
.00658 .4481 m
.01029 .44678 L
s
.00472 .45986 m
.00843 .45855 L
s
.00284 .4717 m
.00656 .4704 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
.636 .78 .324 r
.38696 .73078 .40169 .70185 .42481 .65622 .41063 .68407 Metetra
.636 .78 .324 r
.41063 .68407 .42481 .65622 .44737 .62155 .43396 .63804 Metetra
.763 .919 .548 r
.43396 .63804 .44737 .62155 .46956 .59894 .45674 .60304 Metetra
.848 .995 .833 r
.45674 .60304 .46956 .59894 .49163 .58438 .47918 .58018 Metetra
.787 .903 .923 r
.47918 .58018 .49163 .58438 .51377 .57406 .50153 .56542 Metetra
.723 .819 .924 r
.50153 .56542 .51377 .57406 .53604 .56573 .52395 .55493 Metetra
.69 .776 .917 r
.52395 .55493 .53604 .56573 .55849 .55827 .54652 .54643 Metetra
.675 .757 .912 r
.54652 .54643 .55849 .55827 .58111 .55116 .56927 .53881 Metetra
.669 .748 .91 r
.56927 .53881 .58111 .55116 .60392 .54418 .5922 .53154 Metetra
.666 .744 .909 r
.5922 .53154 .60392 .54418 .62692 .53722 .61533 .52439 Metetra
.664 .743 .908 r
.61533 .52439 .62692 .53722 .65011 .53024 .63864 .51727 Metetra
.664 .742 .908 r
.63864 .51727 .65011 .53024 .67349 .52321 .66216 .51012 Metetra
.664 .742 .908 r
.66216 .51012 .67349 .52321 .69706 .51613 .68587 .50292 Metetra
.664 .741 .908 r
.68587 .50292 .69706 .51613 .72084 .509 .70978 .49568 Metetra
.664 .741 .908 r
.70978 .49568 .72084 .509 .74481 .50181 .7339 .48837 Metetra
.664 .741 .908 r
.7339 .48837 .74481 .50181 .76899 .49455 .75822 .481 Metetra
.664 .741 .908 r
.75822 .481 .76899 .49455 .79338 .48724 .78275 .47357 Metetra
.664 .741 .908 r
.78275 .47357 .79338 .48724 .81797 .47986 .80749 .46607 Metetra
.664 .741 .908 r
.80749 .46607 .81797 .47986 .84277 .47242 .83244 .45851 Metetra
.664 .741 .908 r
.83244 .45851 .84277 .47242 .86778 .46492 .85762 .45089 Metetra
.664 .741 .908 r
.85762 .45089 .86778 .46492 .89301 .45735 .883 .4432 Metetra
.664 .741 .908 r
.883 .4432 .89301 .45735 .91846 .44971 .90862 .43544 Metetra
.664 .741 .908 r
.90862 .43544 .91846 .44971 .94414 .44201 .93445 .42761 Metetra
.664 .741 .908 r
.93445 .42761 .94414 .44201 .97003 .43424 .96052 .41972 Metetra
.754 .913 .537 r
.37214 .74958 .38696 .73078 .41063 .68407 .3959 .71301 Metetra
.636 .78 .324 r
.3959 .71301 .41063 .68407 .43396 .63804 .41979 .66588 Metetra
.636 .78 .324 r
.41979 .66588 .43396 .63804 .45674 .60304 .44332 .61943 Metetra
.763 .919 .548 r
.44332 .61943 .45674 .60304 .47918 .58018 .46633 .5841 Metetra
.848 .995 .833 r
.46633 .5841 .47918 .58018 .50153 .56542 .48902 .56099 Metetra
.787 .903 .923 r
.48902 .56099 .50153 .56542 .52395 .55493 .51165 .54602 Metetra
.723 .819 .924 r
.51165 .54602 .52395 .55493 .54652 .54643 .53437 .53535 Metetra
.69 .776 .917 r
.53437 .53535 .54652 .54643 .56927 .53881 .55725 .52668 Metetra
.675 .757 .912 r
.55725 .52668 .56927 .53881 .5922 .53154 .5803 .51889 Metetra
.669 .748 .91 r
.5803 .51889 .5922 .53154 .61533 .52439 .60356 .51145 Metetra
.666 .744 .909 r
.60356 .51145 .61533 .52439 .63864 .51727 .62701 .50413 Metetra
.664 .743 .908 r
.62701 .50413 .63864 .51727 .66216 .51012 .65065 .49684 Metetra
.664 .742 .908 r
.65065 .49684 .66216 .51012 .68587 .50292 .6745 .48952 Metetra
.664 .742 .908 r
.6745 .48952 .68587 .50292 .70978 .49568 .69855 .48215 Metetra
.664 .741 .908 r
.69855 .48215 .70978 .49568 .7339 .48837 .72281 .47472 Metetra
.664 .741 .908 r
.72281 .47472 .7339 .48837 .75822 .481 .74728 .46723 Metetra
.664 .741 .908 r
.74728 .46723 .75822 .481 .78275 .47357 .77196 .45968 Metetra
.664 .741 .908 r
.77196 .45968 .78275 .47357 .80749 .46607 .79685 .45207 Metetra
.664 .741 .908 r
.79685 .45207 .80749 .46607 .83244 .45851 .82196 .44439 Metetra
.664 .741 .908 r
.82196 .44439 .83244 .45851 .85762 .45089 .84728 .43664 Metetra
.664 .741 .908 r
.84728 .43664 .85762 .45089 .883 .4432 .87283 .42882 Metetra
.664 .741 .908 r
.87283 .42882 .883 .4432 .90862 .43544 .89861 .42094 Metetra
.664 .741 .908 r
.89861 .42094 .90862 .43544 .93445 .42761 .92461 .41298 Metetra
.664 .741 .908 r
.92461 .41298 .93445 .42761 .96052 .41972 .95085 .40496 Metetra
.836 .999 .819 r
.35745 .75582 .37214 .74958 .3959 .71301 .38105 .73174 Metetra
.754 .913 .537 r
.38105 .73174 .3959 .71301 .41979 .66588 .40506 .6948 Metetra
.636 .78 .324 r
.40506 .6948 .41979 .66588 .44332 .61943 .42916 .64724 Metetra
.636 .78 .324 r
.42916 .64724 .44332 .61943 .46633 .5841 .45291 .60038 Metetra
.763 .919 .548 r
.45291 .60038 .46633 .5841 .48902 .56099 .47615 .56471 Metetra
.848 .995 .833 r
.47615 .56471 .48902 .56099 .51165 .54602 .4991 .54134 Metetra
.787 .903 .923 r
.4991 .54134 .51165 .54602 .53437 .53535 .52202 .52616 Metetra
.723 .819 .924 r
.52202 .52616 .53437 .53535 .55725 .52668 .54504 .5153 Metetra
.69 .776 .917 r
.54504 .5153 .55725 .52668 .5803 .51889 .56823 .50645 Metetra
.675 .757 .912 r
.56823 .50645 .5803 .51889 .60356 .51145 .59161 .49848 Metetra
.669 .748 .91 r
.59161 .49848 .60356 .51145 .62701 .50413 .61519 .49087 Metetra
.666 .744 .909 r
.61519 .49087 .62701 .50413 .65065 .49684 .63897 .48338 Metetra
.664 .743 .908 r
.63897 .48338 .65065 .49684 .6745 .48952 .66295 .47591 Metetra
.664 .742 .908 r
.66295 .47591 .6745 .48952 .69855 .48215 .68715 .46841 Metetra
.664 .742 .908 r
.68715 .46841 .69855 .48215 .72281 .47472 .71155 .46086 Metetra
.664 .741 .908 r
.71155 .46086 .72281 .47472 .74728 .46723 .73616 .45325 Metetra
.664 .741 .908 r
.73616 .45325 .74728 .46723 .77196 .45968 .76099 .44557 Metetra
.664 .741 .908 r
.76099 .44557 .77196 .45968 .79685 .45207 .78603 .43784 Metetra
.664 .741 .908 r
.78603 .43784 .79685 .45207 .82196 .44439 .8113 .43003 Metetra
.664 .741 .908 r
.8113 .43003 .82196 .44439 .84728 .43664 .83678 .42216 Metetra
.664 .741 .908 r
.83678 .42216 .84728 .43664 .87283 .42882 .86249 .41421 Metetra
.664 .741 .908 r
.86249 .41421 .87283 .42882 .89861 .42094 .88844 .4062 Metetra
.664 .741 .908 r
.88844 .4062 .89861 .42094 .92461 .41298 .91461 .39811 Metetra
.664 .741 .908 r
.91461 .39811 .92461 .41298 .95085 .40496 .94101 .38995 Metetra
.78 .925 .931 r
.34283 .75317 .35745 .75582 .38105 .73174 .36629 .73782 Metetra
.836 .999 .819 r
.36629 .73782 .38105 .73174 .40506 .6948 .39018 .71346 Metetra
.754 .913 .537 r
.39018 .71346 .40506 .6948 .42916 .64724 .41444 .67615 Metetra
.636 .78 .324 r
.41444 .67615 .42916 .64724 .45291 .60038 .43876 .62815 Metetra
.636 .78 .324 r
.43876 .62815 .45291 .60038 .47615 .56471 .46272 .58087 Metetra
.763 .919 .548 r
.46272 .58087 .47615 .56471 .4991 .54134 .4862 .54485 Metetra
.848 .995 .833 r
.4862 .54485 .4991 .54134 .52202 .52616 .50942 .52122 Metetra
.787 .903 .923 r
.50942 .52122 .52202 .52616 .54504 .5153 .53263 .50582 Metetra
.723 .819 .924 r
.53263 .50582 .54504 .5153 .56823 .50645 .55596 .49476 Metetra
.69 .776 .917 r
.55596 .49476 .56823 .50645 .59161 .49848 .57947 .48573 Metetra
.675 .757 .912 r
.57947 .48573 .59161 .49848 .61519 .49087 .60318 .47758 Metetra
.669 .748 .91 r
.60318 .47758 .61519 .49087 .63897 .48338 .6271 .46979 Metetra
.666 .744 .909 r
.6271 .46979 .63897 .48338 .66295 .47591 .65122 .46212 Metetra
.664 .743 .908 r
.65122 .46212 .66295 .47591 .68715 .46841 .67556 .45447 Metetra
.664 .742 .908 r
.67556 .45447 .68715 .46841 .71155 .46086 .7001 .44678 Metetra
.664 .742 .908 r
.7001 .44678 .71155 .46086 .73616 .45325 .72486 .43904 Metetra
.664 .741 .908 r
.72486 .43904 .73616 .45325 .76099 .44557 .74984 .43124 Metetra
.664 .741 .908 r
.74984 .43124 .76099 .44557 .78603 .43784 .77504 .42337 Metetra
.664 .741 .908 r
.77504 .42337 .78603 .43784 .8113 .43003 .80046 .41544 Metetra
.664 .741 .908 r
.80046 .41544 .8113 .43003 .83678 .42216 .82611 .40743 Metetra
.664 .741 .908 r
.82611 .40743 .83678 .42216 .86249 .41421 .85199 .39936 Metetra
.664 .741 .908 r
.85199 .39936 .86249 .41421 .88844 .4062 .87809 .39121 Metetra
.664 .741 .908 r
.87809 .39121 .88844 .4062 .91461 .39811 .90444 .38299 Metetra
.664 .741 .908 r
.90444 .38299 .91461 .39811 .94101 .38995 .93102 .3747 Metetra
.714 .844 .944 r
.32817 .74559 .34283 .75317 .36629 .73782 .35158 .73495 Metetra
.78 .925 .931 r
.35158 .73495 .36629 .73782 .39018 .71346 .37535 .71938 Metetra
.836 .999 .819 r
.37535 .71938 .39018 .71346 .41444 .67615 .39953 .69472 Metetra
.754 .913 .537 r
.39953 .69472 .41444 .67615 .43876 .62815 .42405 .65703 Metetra
.636 .78 .324 r
.42405 .65703 .43876 .62815 .46272 .58087 .4486 .60859 Metetra
.636 .78 .324 r
.4486 .60859 .46272 .58087 .4862 .54485 .47278 .56088 Metetra
.763 .919 .548 r
.47278 .56088 .4862 .54485 .50942 .52122 .4965 .52452 Metetra
.848 .995 .833 r
.4965 .52452 .50942 .52122 .53263 .50582 .52 .50061 Metetra
.787 .903 .923 r
.52 .50061 .53263 .50582 .55596 .49476 .54351 .48499 Metetra
.723 .819 .924 r
.54351 .48499 .55596 .49476 .57947 .48573 .56716 .47373 Metetra
.69 .776 .917 r
.56716 .47373 .57947 .48573 .60318 .47758 .591 .46451 Metetra
.675 .757 .912 r
.591 .46451 .60318 .47758 .6271 .46979 .61505 .45617 Metetra
.669 .748 .91 r
.61505 .45617 .6271 .46979 .65122 .46212 .63931 .44819 Metetra
.666 .744 .909 r
.63931 .44819 .65122 .46212 .67556 .45447 .66378 .44033 Metetra
.664 .743 .908 r
.66378 .44033 .67556 .45447 .7001 .44678 .68847 .43249 Metetra
.664 .742 .908 r
.68847 .43249 .7001 .44678 .72486 .43904 .71339 .42461 Metetra
.664 .742 .908 r
.71339 .42461 .72486 .43904 .74984 .43124 .73852 .41667 Metetra
.664 .741 .908 r
.73852 .41667 .74984 .43124 .77504 .42337 .76387 .40867 Metetra
.664 .741 .908 r
.76387 .40867 .77504 .42337 .80046 .41544 .78945 .40061 Metetra
.664 .741 .908 r
.78945 .40061 .80046 .41544 .82611 .40743 .81526 .39247 Metetra
.664 .741 .908 r
.81526 .39247 .82611 .40743 .85199 .39936 .8413 .38426 Metetra
.664 .741 .908 r
.8413 .38426 .85199 .39936 .87809 .39121 .86758 .37598 Metetra
.664 .741 .908 r
.86758 .37598 .87809 .39121 .90444 .38299 .8941 .36762 Metetra
.664 .741 .908 r
.8941 .36762 .90444 .38299 .93102 .3747 .92085 .35918 Metetra
.677 .799 .94 r
.31338 .73554 .32817 .74559 .35158 .73495 .33682 .7271 Metetra
.714 .844 .944 r
.33682 .7271 .35158 .73495 .37535 .71938 .36056 .71627 Metetra
.78 .925 .931 r
.36056 .71627 .37535 .71938 .39953 .69472 .38464 .70047 Metetra
.836 .999 .819 r
.38464 .70047 .39953 .69472 .42405 .65703 .40912 .67551 Metetra
.754 .913 .537 r
.40912 .67551 .42405 .65703 .4486 .60859 .43391 .63744 Metetra
.636 .78 .324 r
.43391 .63744 .4486 .60859 .47278 .56088 .45868 .58854 Metetra
.636 .78 .324 r
.45868 .58854 .47278 .56088 .4965 .52452 .48308 .5404 Metetra
.763 .919 .548 r
.48308 .5404 .4965 .52452 .52 .50061 .50705 .50368 Metetra
.848 .995 .833 r
.50705 .50368 .52 .50061 .54351 .48499 .53083 .47949 Metetra
.787 .903 .923 r
.53083 .47949 .54351 .48499 .56716 .47373 .55465 .46364 Metetra
.723 .819 .924 r
.55465 .46364 .56716 .47373 .591 .46451 .57862 .45217 Metetra
.69 .776 .917 r
.57862 .45217 .591 .46451 .61505 .45617 .60281 .44275 Metetra
.675 .757 .912 r
.60281 .44275 .61505 .45617 .63931 .44819 .6272 .43423 Metetra
.669 .748 .91 r
.6272 .43423 .63931 .44819 .66378 .44033 .65182 .42605 Metetra
.666 .744 .909 r
.65182 .42605 .66378 .44033 .68847 .43249 .67666 .418 Metetra
.664 .743 .908 r
.67666 .418 .68847 .43249 .71339 .42461 .70172 .40995 Metetra
.664 .742 .908 r
.70172 .40995 .71339 .42461 .73852 .41667 .727 .40187 Metetra
.664 .742 .908 r
.727 .40187 .73852 .41667 .76387 .40867 .75252 .39374 Metetra
.664 .741 .908 r
.75252 .39374 .76387 .40867 .78945 .40061 .77826 .38553 Metetra
.664 .741 .908 r
.77826 .38553 .78945 .40061 .81526 .39247 .80423 .37726 Metetra
.664 .741 .908 r
.80423 .37726 .81526 .39247 .8413 .38426 .83044 .36891 Metetra
.664 .741 .908 r
.83044 .36891 .8413 .38426 .86758 .37598 .85689 .36049 Metetra
.664 .741 .908 r
.85689 .36049 .86758 .37598 .8941 .36762 .88358 .35199 Metetra
.664 .741 .908 r
.88358 .35199 .8941 .36762 .92085 .35918 .91052 .34341 Metetra
.66 .779 .937 r
.29839 .72425 .31338 .73554 .33682 .7271 .32192 .71676 Metetra
.677 .799 .94 r
.32192 .71676 .33682 .7271 .36056 .71627 .34569 .70815 Metetra
.714 .844 .944 r
.34569 .70815 .36056 .71627 .38464 .70047 .36976 .69711 Metetra
.78 .925 .931 r
.36976 .69711 .38464 .70047 .40912 .67551 .39417 .68107 Metetra
.836 .999 .819 r
.39417 .68107 .40912 .67551 .43391 .63744 .41895 .65581 Metetra
.754 .913 .537 r
.41895 .65581 .43391 .63744 .45868 .58854 .44401 .61735 Metetra
.636 .78 .324 r
.44401 .61735 .45868 .58854 .48308 .5404 .46902 .56799 Metetra
.636 .78 .324 r
.46902 .56799 .48308 .5404 .50705 .50368 .49365 .51941 Metetra
.763 .919 .548 r
.49365 .51941 .50705 .50368 .53083 .47949 .51786 .48232 Metetra
.848 .995 .833 r
.51786 .48232 .53083 .47949 .55465 .46364 .54193 .45785 Metetra
.787 .903 .923 r
.54193 .45785 .55465 .46364 .57862 .45217 .56607 .44176 Metetra
.723 .819 .924 r
.56607 .44176 .57862 .45217 .60281 .44275 .59038 .43008 Metetra
.69 .776 .917 r
.59038 .43008 .60281 .44275 .6272 .43423 .61491 .42046 Metetra
.675 .757 .912 r
.61491 .42046 .6272 .43423 .65182 .42605 .63967 .41173 Metetra
.669 .748 .91 r
.63967 .41173 .65182 .42605 .67666 .418 .66465 .40335 Metetra
.666 .744 .909 r
.66465 .40335 .67666 .418 .70172 .40995 .68986 .39509 Metetra
.664 .743 .908 r
.68986 .39509 .70172 .40995 .727 .40187 .7153 .38684 Metetra
.664 .742 .908 r
.7153 .38684 .727 .40187 .75252 .39374 .74097 .37856 Metetra
.664 .742 .908 r
.74097 .37856 .75252 .39374 .77826 .38553 .76688 .37021 Metetra
.664 .741 .908 r
.76688 .37021 .77826 .38553 .80423 .37726 .79302 .36179 Metetra
.664 .741 .908 r
.79302 .36179 .80423 .37726 .83044 .36891 .8194 .3533 Metetra
.664 .741 .908 r
.8194 .3533 .83044 .36891 .85689 .36049 .84602 .34473 Metetra
.664 .741 .908 r
.84602 .34473 .85689 .36049 .88358 .35199 .87289 .33609 Metetra
.664 .741 .908 r
.87289 .33609 .88358 .35199 .91052 .34341 .90001 .32736 Metetra
.652 .769 .935 r
.28318 .71229 .29839 .72425 .32192 .71676 .30681 .70517 Metetra
.66 .779 .937 r
.30681 .70517 .32192 .71676 .34569 .70815 .33067 .6975 Metetra
.677 .799 .94 r
.33067 .6975 .34569 .70815 .36976 .69711 .35479 .6887 Metetra
.714 .844 .944 r
.35479 .6887 .36976 .69711 .39417 .68107 .3792 .67746 Metetra
.78 .925 .931 r
.3792 .67746 .39417 .68107 .41895 .65581 .40395 .66117 Metetra
.836 .999 .819 r
.40395 .66117 .41895 .65581 .44401 .61735 .42904 .63561 Metetra
.754 .913 .537 r
.42904 .63561 .44401 .61735 .46902 .56799 .45438 .59674 Metetra
.636 .78 .324 r
.45438 .59674 .46902 .56799 .49365 .51941 .47962 .54691 Metetra
.636 .78 .324 r
.47962 .54691 .49365 .51941 .51786 .48232 .50448 .49788 Metetra
.763 .919 .548 r
.50448 .49788 .51786 .48232 .54193 .45785 .52895 .46042 Metetra
.848 .995 .833 r
.52895 .46042 .54193 .45785 .56607 .44176 .55331 .43566 Metetra
.787 .903 .923 r
.55331 .43566 .56607 .44176 .59038 .43008 .57777 .41932 Metetra
.723 .819 .924 r
.57777 .41932 .59038 .43008 .61491 .42046 .60243 .40742 Metetra
.69 .776 .917 r
.60243 .40742 .61491 .42046 .63967 .41173 .62732 .39759 Metetra
.675 .757 .912 r
.62732 .39759 .63967 .41173 .66465 .40335 .65245 .38865 Metetra
.669 .748 .91 r
.65245 .38865 .66465 .40335 .68986 .39509 .67781 .38007 Metetra
.666 .744 .909 r
.67781 .38007 .68986 .39509 .7153 .38684 .7034 .3716 Metetra
.664 .743 .908 r
.7034 .3716 .7153 .38684 .74097 .37856 .72924 .36314 Metetra
.664 .742 .908 r
.72924 .36314 .74097 .37856 .76688 .37021 .7553 .35463 Metetra
.664 .742 .908 r
.7553 .35463 .76688 .37021 .79302 .36179 .78161 .34607 Metetra
.664 .741 .908 r
.78161 .34607 .79302 .36179 .8194 .3533 .80816 .33743 Metetra
.664 .741 .908 r
.80816 .33743 .8194 .3533 .84602 .34473 .83496 .32871 Metetra
.664 .741 .908 r
.83496 .32871 .84602 .34473 .87289 .33609 .86201 .31992 Metetra
.664 .741 .908 r
.86201 .31992 .87289 .33609 .90001 .32736 .88931 .31104 Metetra
.649 .765 .934 r
.26774 .69993 .28318 .71229 .30681 .70517 .29148 .69289 Metetra
.652 .769 .935 r
.29148 .69289 .30681 .70517 .33067 .6975 .31545 .68559 Metetra
.66 .779 .937 r
.31545 .68559 .33067 .6975 .35479 .6887 .33966 .67774 Metetra
.677 .799 .94 r
.33966 .67774 .35479 .6887 .3792 .67746 .36413 .66875 Metetra
.714 .844 .944 r
.36413 .66875 .3792 .67746 .40395 .66117 .38889 .65729 Metetra
.78 .925 .931 r
.38889 .65729 .40395 .66117 .42904 .63561 .41398 .64075 Metetra
.836 .999 .819 r
.41398 .64075 .42904 .63561 .45438 .59674 .4394 .61487 Metetra
.754 .913 .537 r
.4394 .61487 .45438 .59674 .47962 .54691 .46501 .57559 Metetra
.636 .78 .324 r
.46501 .57559 .47962 .54691 .50448 .49788 .4905 .52529 Metetra
.636 .78 .324 r
.4905 .52529 .50448 .49788 .52895 .46042 .51558 .4758 Metetra
.763 .919 .548 r
.51558 .4758 .52895 .46042 .55331 .43566 .54032 .43797 Metetra
.848 .995 .833 r
.54032 .43797 .55331 .43566 .57777 .41932 .56498 .4129 Metetra
.787 .903 .923 r
.56498 .4129 .57777 .41932 .60243 .40742 .58978 .39632 Metetra
.723 .819 .924 r
.58978 .39632 .60243 .40742 .62732 .39759 .6148 .38419 Metetra
.69 .776 .917 r
.6148 .38419 .62732 .39759 .65245 .38865 .64006 .37413 Metetra
.675 .757 .912 r
.64006 .37413 .65245 .38865 .67781 .38007 .66556 .36498 Metetra
.669 .748 .91 r
.66556 .36498 .67781 .38007 .7034 .3716 .69131 .35618 Metetra
.666 .744 .909 r
.69131 .35618 .7034 .3716 .72924 .36314 .7173 .34749 Metetra
.664 .743 .908 r
.7173 .34749 .72924 .36314 .7553 .35463 .74353 .33881 Metetra
.664 .742 .908 r
.74353 .33881 .7553 .35463 .78161 .34607 .77001 .33008 Metetra
.664 .742 .908 r
.77001 .33008 .78161 .34607 .80816 .33743 .79674 .32128 Metetra
.664 .741 .908 r
.79674 .32128 .80816 .33743 .83496 .32871 .82371 .31241 Metetra
.664 .741 .908 r
.82371 .31241 .83496 .32871 .86201 .31992 .85094 .30347 Metetra
.664 .741 .908 r
.85094 .30347 .86201 .31992 .88931 .31104 .87843 .29443 Metetra
.647 .764 .934 r
.25204 .68726 .26774 .69993 .29148 .69289 .2759 .68021 Metetra
.649 .765 .934 r
.2759 .68021 .29148 .69289 .31545 .68559 .29999 .67299 Metetra
.652 .769 .935 r
.29999 .67299 .31545 .68559 .33966 .67774 .32432 .6655 Metetra
.66 .779 .937 r
.32432 .6655 .33966 .67774 .36413 .66875 .34889 .65747 Metetra
.677 .799 .94 r
.34889 .65747 .36413 .66875 .38889 .65729 .37372 .64827 Metetra
.714 .844 .944 r
.37372 .64827 .38889 .65729 .41398 .64075 .39884 .63659 Metetra
.78 .925 .931 r
.39884 .63659 .41398 .64075 .4394 .61487 .42428 .61979 Metetra
.836 .999 .819 r
.42428 .61979 .4394 .61487 .46501 .57559 .45003 .59358 Metetra
.754 .913 .537 r
.45003 .59358 .46501 .57559 .4905 .52529 .47593 .55388 Metetra
.636 .78 .324 r
.47593 .55388 .4905 .52529 .51558 .4758 .50166 .5031 Metetra
.636 .78 .324 r
.50166 .5031 .51558 .4758 .54032 .43797 .52698 .45314 Metetra
.763 .919 .548 r
.52698 .45314 .54032 .43797 .56498 .4129 .55198 .41493 Metetra
.848 .995 .833 r
.55198 .41493 .56498 .4129 .58978 .39632 .57696 .38956 Metetra
.787 .903 .923 r
.57696 .38956 .58978 .39632 .6148 .38419 .6021 .37271 Metetra
.723 .819 .924 r
.6021 .37271 .6148 .38419 .64006 .37413 .62748 .36035 Metetra
.69 .776 .917 r
.62748 .36035 .64006 .37413 .66556 .36498 .65312 .35007 Metetra
.675 .757 .912 r
.65312 .35007 .66556 .36498 .69131 .35618 .67902 .34069 Metetra
.669 .748 .91 r
.67902 .34069 .69131 .35618 .7173 .34749 .70517 .33166 Metetra
.666 .744 .909 r
.70517 .33166 .7173 .34749 .74353 .33881 .73157 .32274 Metetra
.664 .743 .908 r
.73157 .32274 .74353 .33881 .77001 .33008 .75821 .31383 Metetra
.664 .742 .908 r
.75821 .31383 .77001 .33008 .79674 .32128 .78512 .30487 Metetra
.664 .742 .908 r
.78512 .30487 .79674 .32128 .82371 .31241 .81227 .29584 Metetra
.664 .741 .908 r
.81227 .29584 .82371 .31241 .85094 .30347 .83968 .28673 Metetra
.664 .741 .908 r
.83968 .28673 .85094 .30347 .87843 .29443 .86736 .27753 Metetra
.646 .763 .934 r
.23607 .67434 .25204 .68726 .2759 .68021 .26007 .66721 Metetra
.647 .764 .934 r
.26007 .66721 .2759 .68021 .29999 .67299 .28429 .65997 Metetra
.649 .765 .934 r
.28429 .65997 .29999 .67299 .32432 .6655 .30873 .65256 Metetra
.652 .769 .935 r
.30873 .65256 .32432 .6655 .34889 .65747 .33342 .64488 Metetra
.66 .779 .937 r
.33342 .64488 .34889 .65747 .37372 .64827 .35836 .63665 Metetra
.677 .799 .94 r
.35836 .63665 .37372 .64827 .39884 .63659 .38356 .62724 Metetra
.714 .844 .944 r
.38356 .62724 .39884 .63659 .42428 .61979 .40905 .61533 Metetra
.78 .925 .931 r
.40905 .61533 .42428 .61979 .45003 .59358 .43486 .59826 Metetra
.836 .999 .819 r
.43486 .59826 .45003 .59358 .47593 .55388 .46094 .57171 Metetra
.754 .913 .537 r
.46094 .57171 .47593 .55388 .50166 .5031 .48714 .53159 Metetra
.636 .78 .324 r
.48714 .53159 .50166 .5031 .52698 .45314 .51312 .48032 Metetra
.636 .78 .324 r
.51312 .48032 .52698 .45314 .55198 .41493 .53868 .42989 Metetra
.763 .919 .548 r
.53868 .42989 .55198 .41493 .57696 .38956 .56396 .39129 Metetra
.848 .995 .833 r
.56396 .39129 .57696 .38956 .6021 .37271 .58925 .3656 Metetra
.787 .903 .923 r
.58925 .3656 .6021 .37271 .62748 .36035 .61474 .34849 Metetra
.723 .819 .924 r
.61474 .34849 .62748 .36035 .65312 .35007 .6405 .33588 Metetra
.69 .776 .917 r
.6405 .33588 .65312 .35007 .67902 .34069 .66653 .32536 Metetra
.675 .757 .912 r
.66653 .32536 .67902 .34069 .70517 .33166 .69283 .31575 Metetra
.669 .748 .91 r
.69283 .31575 .70517 .33166 .73157 .32274 .7194 .30649 Metetra
.666 .744 .909 r
.7194 .30649 .73157 .32274 .75821 .31383 .74621 .29733 Metetra
.664 .743 .908 r
.74621 .29733 .75821 .31383 .78512 .30487 .77329 .28818 Metetra
.664 .742 .908 r
.77329 .28818 .78512 .30487 .81227 .29584 .80063 .27897 Metetra
.664 .742 .908 r
.80063 .27897 .81227 .29584 .83968 .28673 .82823 .2697 Metetra
.664 .741 .908 r
.82823 .2697 .83968 .28673 .86736 .27753 .85609 .26034 Metetra
.646 .763 .934 r
.21984 .66119 .23607 .67434 .26007 .66721 .24396 .65395 Metetra
.646 .763 .934 r
.24396 .65395 .26007 .66721 .28429 .65997 .26831 .64662 Metetra
.647 .764 .934 r
.26831 .64662 .28429 .65997 .30873 .65256 .29289 .63919 Metetra
.649 .765 .934 r
.29289 .63919 .30873 .65256 .33342 .64488 .31771 .63159 Metetra
.652 .769 .935 r
.31771 .63159 .33342 .64488 .35836 .63665 .34277 .62371 Metetra
.66 .779 .937 r
.34277 .62371 .35836 .63665 .38356 .62724 .36808 .61526 Metetra
.677 .799 .94 r
.36808 .61526 .38356 .62724 .40905 .61533 .39367 .60564 Metetra
.714 .844 .944 r
.39367 .60564 .40905 .61533 .43486 .59826 .41954 .59349 Metetra
.78 .925 .931 r
.41954 .59349 .43486 .59826 .46094 .57171 .44572 .57615 Metetra
.836 .999 .819 r
.44572 .57615 .46094 .57171 .48714 .53159 .47215 .54925 Metetra
.754 .913 .537 r
.47215 .54925 .48714 .53159 .51312 .48032 .49865 .50869 Metetra
.636 .78 .324 r
.49865 .50869 .51312 .48032 .53868 .42989 .52489 .45693 Metetra
.636 .78 .324 r
.52489 .45693 .53868 .42989 .56396 .39129 .55069 .40602 Metetra
.763 .919 .548 r
.55069 .40602 .56396 .39129 .58925 .3656 .57624 .36702 Metetra
.848 .995 .833 r
.57624 .36702 .58925 .3656 .61474 .34849 .60186 .34101 Metetra
.787 .903 .923 r
.60186 .34101 .61474 .34849 .6405 .33588 .62772 .32362 Metetra
.723 .819 .924 r
.62772 .32362 .6405 .33588 .66653 .32536 .65387 .31076 Metetra
.69 .776 .917 r
.65387 .31076 .66653 .32536 .69283 .31575 .68031 .29999 Metetra
.675 .757 .912 r
.68031 .29999 .69283 .31575 .7194 .30649 .70702 .29014 Metetra
.669 .748 .91 r
.70702 .29014 .7194 .30649 .74621 .29733 .73401 .28063 Metetra
.666 .744 .909 r
.73401 .28063 .74621 .29733 .77329 .28818 .76126 .27123 Metetra
.664 .743 .908 r
.76126 .27123 .77329 .28818 .80063 .27897 .78878 .26183 Metetra
.664 .742 .908 r
.78878 .26183 .80063 .27897 .82823 .2697 .81656 .25237 Metetra
.664 .742 .908 r
.81656 .25237 .82823 .2697 .85609 .26034 .84462 .24284 Metetra
.646 .763 .934 r
.20332 .6478 .21984 .66119 .24396 .65395 .22758 .64044 Metetra
.646 .763 .934 r
.22758 .64044 .24396 .65395 .26831 .64662 .25206 .633 Metetra
.646 .763 .934 r
.25206 .633 .26831 .64662 .29289 .63919 .27678 .62548 Metetra
.647 .764 .934 r
.27678 .62548 .29289 .63919 .31771 .63159 .30173 .61784 Metetra
.649 .765 .934 r
.30173 .61784 .31771 .63159 .34277 .62371 .32692 .61004 Metetra
.652 .769 .935 r
.32692 .61004 .34277 .62371 .36808 .61526 .35237 .60195 Metetra
.66 .779 .937 r
.35237 .60195 .36808 .61526 .39367 .60564 .37807 .59329 Metetra
.677 .799 .94 r
.37807 .59329 .39367 .60564 .41954 .59349 .40405 .58344 Metetra
.714 .844 .944 r
.40405 .58344 .41954 .59349 .44572 .57615 .43033 .57105 Metetra
.78 .925 .931 r
.43033 .57105 .44572 .57615 .47215 .54925 .45689 .55342 Metetra
.836 .999 .819 r
.45689 .55342 .47215 .54925 .49865 .50869 .48367 .52617 Metetra
.754 .913 .537 r
.48367 .52617 .49865 .50869 .52489 .45693 .51048 .48517 Metetra
.636 .78 .324 r
.51048 .48517 .52489 .45693 .55069 .40602 .53697 .4329 Metetra
.636 .78 .324 r
.53697 .4329 .55069 .40602 .57624 .36702 .56302 .3815 Metetra
.763 .919 .548 r
.56302 .3815 .57624 .36702 .60186 .34101 .58886 .34209 Metetra
.848 .995 .833 r
.58886 .34209 .60186 .34101 .62772 .32362 .61482 .31575 Metetra
.787 .903 .923 r
.61482 .31575 .62772 .32362 .65387 .31076 .64105 .29808 Metetra
.723 .819 .924 r
.64105 .29808 .65387 .31076 .68031 .29999 .6676 .28495 Metetra
.69 .776 .917 r
.6676 .28495 .68031 .29999 .70702 .29014 .69445 .27394 Metetra
.675 .757 .912 r
.69445 .27394 .70702 .29014 .73401 .28063 .7216 .26383 Metetra
.669 .748 .91 r
.7216 .26383 .73401 .28063 .76126 .27123 .74902 .25407 Metetra
.666 .744 .909 r
.74902 .25407 .76126 .27123 .78878 .26183 .77672 .24441 Metetra
.664 .743 .908 r
.77672 .24441 .78878 .26183 .81656 .25237 .80469 .23475 Metetra
.664 .742 .908 r
.80469 .23475 .81656 .25237 .84462 .24284 .83294 .22503 Metetra
.646 .762 .934 r
.18652 .63417 .20332 .6478 .22758 .64044 .21091 .62669 Metetra
.646 .763 .934 r
.21091 .62669 .22758 .64044 .25206 .633 .23553 .61913 Metetra
.646 .763 .934 r
.23553 .61913 .25206 .633 .27678 .62548 .26038 .61149 Metetra
.646 .763 .934 r
.26038 .61149 .27678 .62548 .30173 .61784 .28547 .60376 Metetra
.647 .764 .934 r
.28547 .60376 .30173 .61784 .32692 .61004 .31081 .59592 Metetra
.649 .765 .934 r
.31081 .59592 .32692 .61004 .35237 .60195 .33639 .5879 Metetra
.652 .769 .935 r
.33639 .5879 .35237 .60195 .37807 .59329 .36224 .57959 Metetra
.66 .779 .937 r
.36224 .57959 .37807 .59329 .40405 .58344 .38834 .57071 Metetra
.677 .799 .94 r
.38834 .57071 .40405 .58344 .43033 .57105 .41473 .56063 Metetra
.714 .844 .944 r
.41473 .56063 .43033 .57105 .45689 .55342 .44141 .54798 Metetra
.78 .925 .931 r
.44141 .54798 .45689 .55342 .48367 .52617 .46837 .53005 Metetra
.836 .999 .819 r
.46837 .53005 .48367 .52617 .51048 .48517 .49552 .50244 Metetra
.754 .913 .537 r
.49552 .50244 .51048 .48517 .53697 .4329 .52264 .46099 Metetra
.636 .78 .324 r
.52264 .46099 .53697 .4329 .56302 .3815 .5494 .4082 Metetra
.636 .78 .324 r
.5494 .4082 .56302 .3815 .58886 .34209 .5757 .35631 Metetra
.763 .919 .548 r
.5757 .35631 .58886 .34209 .61482 .31575 .60183 .31649 Metetra
.848 .995 .833 r
.60183 .31649 .61482 .31575 .64105 .29808 .62812 .28981 Metetra
.787 .903 .923 r
.62812 .28981 .64105 .29808 .6676 .28495 .65474 .27184 Metetra
.723 .819 .924 r
.65474 .27184 .6676 .28495 .69445 .27394 .6817 .25844 Metetra
.69 .776 .917 r
.6817 .25844 .69445 .27394 .7216 .26383 .70899 .24716 Metetra
.675 .757 .912 r
.70899 .24716 .7216 .26383 .74902 .25407 .73657 .23679 Metetra
.669 .748 .91 r
.73657 .23679 .74902 .25407 .77672 .24441 .76445 .22677 Metetra
.666 .744 .909 r
.76445 .22677 .77672 .24441 .80469 .23475 .79261 .21684 Metetra
.664 .743 .908 r
.79261 .21684 .80469 .23475 .83294 .22503 .82106 .20691 Metetra
.646 .762 .934 r
.16943 .6203 .18652 .63417 .21091 .62669 .19395 .61269 Metetra
.646 .762 .934 r
.19395 .61269 .21091 .62669 .23553 .61913 .2187 .605 Metetra
.646 .763 .934 r
.2187 .605 .23553 .61913 .26038 .61149 .24369 .59723 Metetra
.646 .763 .934 r
.24369 .59723 .26038 .61149 .28547 .60376 .26893 .58939 Metetra
.646 .763 .934 r
.26893 .58939 .28547 .60376 .31081 .59592 .29441 .58144 Metetra
.647 .764 .934 r
.29441 .58144 .31081 .59592 .33639 .5879 .32014 .57338 Metetra
.649 .765 .934 r
.32014 .57338 .33639 .5879 .36224 .57959 .34613 .56514 Metetra
.652 .769 .935 r
.34613 .56514 .36224 .57959 .38834 .57071 .37238 .55661 Metetra
.66 .779 .937 r
.37238 .55661 .38834 .57071 .41473 .56063 .3989 .54749 Metetra
.677 .799 .94 r
.3989 .54749 .41473 .56063 .44141 .54798 .42571 .53716 Metetra
.714 .844 .944 r
.42571 .53716 .44141 .54798 .46837 .53005 .45281 .52425 Metetra
.78 .925 .931 r
.45281 .52425 .46837 .53005 .49552 .50244 .48017 .50602 Metetra
.836 .999 .819 r
.48017 .50602 .49552 .50244 .52264 .46099 .5077 .47804 Metetra
.754 .913 .537 r
.5077 .47804 .52264 .46099 .5494 .4082 .53514 .43612 Metetra
.636 .78 .324 r
.53514 .43612 .5494 .4082 .5757 .35631 .56217 .38281 Metetra
.636 .78 .324 r
.56217 .38281 .5757 .35631 .60183 .31649 .58872 .33042 Metetra
.763 .919 .548 r
.58872 .33042 .60183 .31649 .62812 .28981 .61515 .29017 Metetra
.848 .995 .833 r
.61515 .29017 .62812 .28981 .65474 .27184 .6418 .26315 Metetra
.787 .903 .923 r
.6418 .26315 .65474 .27184 .6817 .25844 .66881 .24488 Metetra
.723 .819 .924 r
.66881 .24488 .6817 .25844 .70899 .24716 .6962 .2312 Metetra
.69 .776 .917 r
.6962 .2312 .70899 .24716 .73657 .23679 .72393 .21964 Metetra
.675 .757 .912 r
.72393 .21964 .73657 .23679 .76445 .22677 .75197 .209 Metetra
.669 .748 .91 r
.75197 .209 .76445 .22677 .79261 .21684 .78032 .1987 Metetra
.666 .744 .909 r
.78032 .1987 .79261 .21684 .82106 .20691 .80895 .18849 Metetra
.646 .762 .934 r
.15203 .60619 .16943 .6203 .19395 .61269 .17668 .59844 Metetra
.646 .762 .934 r
.17668 .59844 .19395 .61269 .2187 .605 .20158 .59062 Metetra
.646 .762 .934 r
.20158 .59062 .2187 .605 .24369 .59723 .22671 .58272 Metetra
.646 .763 .934 r
.22671 .58272 .24369 .59723 .26893 .58939 .25209 .57473 Metetra
.646 .763 .934 r
.25209 .57473 .26893 .58939 .29441 .58144 .27771 .56666 Metetra
.646 .763 .934 r
.27771 .56666 .29441 .58144 .32014 .57338 .3036 .5585 Metetra
.647 .764 .934 r
.3036 .5585 .32014 .57338 .34613 .56514 .32974 .55021 Metetra
.649 .765 .934 r
.32974 .55021 .34613 .56514 .37238 .55661 .35614 .54174 Metetra
.652 .769 .935 r
.35614 .54174 .37238 .55661 .3989 .54749 .38281 .53297 Metetra
.66 .779 .937 r
.38281 .53297 .3989 .54749 .42571 .53716 .40977 .52361 Metetra
.677 .799 .94 r
.40977 .52361 .42571 .53716 .45281 .52425 .43701 .51303 Metetra
.714 .844 .944 r
.43701 .51303 .45281 .52425 .48017 .50602 .46454 .49984 Metetra
.78 .925 .931 r
.46454 .49984 .48017 .50602 .5077 .47804 .49232 .4813 Metetra
.836 .999 .819 r
.49232 .4813 .5077 .47804 .53514 .43612 .52024 .45293 Metetra
.754 .913 .537 r
.52024 .45293 .53514 .43612 .56217 .38281 .54801 .41054 Metetra
.636 .78 .324 r
.54801 .41054 .56217 .38281 .58872 .33042 .5753 .3567 Metetra
.636 .78 .324 r
.5753 .3567 .58872 .33042 .61515 .29017 .60212 .3038 Metetra
.763 .919 .548 r
.60212 .3038 .61515 .29017 .6418 .26315 .62885 .26312 Metetra
.848 .995 .833 r
.62885 .26312 .6418 .26315 .66881 .24488 .65586 .23574 Metetra
.787 .903 .923 r
.65586 .23574 .66881 .24488 .6962 .2312 .68328 .21716 Metetra
.723 .819 .924 r
.68328 .21716 .6962 .2312 .72393 .21964 .71111 .20318 Metetra
.69 .776 .917 r
.71111 .20318 .72393 .21964 .75197 .209 .73929 .19134 Metetra
.675 .757 .912 r
.73929 .19134 .75197 .209 .78032 .1987 .76781 .18041 Metetra
.669 .748 .91 r
.76781 .18041 .78032 .1987 .80895 .18849 .79663 .16982 Metetra
.646 .762 .934 r
.13432 .59183 .15203 .60619 .17668 .59844 .15911 .58394 Metetra
.646 .762 .934 r
.15911 .58394 .17668 .59844 .20158 .59062 .18414 .57597 Metetra
.646 .762 .934 r
.18414 .57597 .20158 .59062 .22671 .58272 .20942 .56793 Metetra
.646 .762 .934 r
.20942 .56793 .22671 .58272 .25209 .57473 .23494 .55981 Metetra
.646 .763 .934 r
.23494 .55981 .25209 .57473 .27771 .56666 .26072 .5516 Metetra
.646 .763 .934 r
.26072 .5516 .27771 .56666 .3036 .5585 .28675 .5433 Metetra
.646 .763 .934 r
.28675 .5433 .3036 .5585 .32974 .55021 .31304 .5349 Metetra
.647 .764 .934 r
.31304 .5349 .32974 .55021 .35614 .54174 .33961 .52637 Metetra
.649 .765 .934 r
.33961 .52637 .35614 .54174 .38281 .53297 .36644 .51767 Metetra
.652 .769 .935 r
.36644 .51767 .38281 .53297 .40977 .52361 .39355 .50865 Metetra
.66 .779 .937 r
.39355 .50865 .40977 .52361 .43701 .51303 .42094 .49904 Metetra
.677 .799 .94 r
.42094 .49904 .43701 .51303 .46454 .49984 .44863 .48819 Metetra
.714 .844 .944 r
.44863 .48819 .46454 .49984 .49232 .4813 .47661 .47471 Metetra
.78 .925 .931 r
.47661 .47471 .49232 .4813 .52024 .45293 .50483 .45585 Metetra
.836 .999 .819 r
.50483 .45585 .52024 .45293 .54801 .41054 .53314 .42709 Metetra
.754 .913 .537 r
.53314 .42709 .54801 .41054 .5753 .3567 .56125 .38422 Metetra
.636 .78 .324 r
.56125 .38422 .5753 .3567 .60212 .3038 .58882 .32983 Metetra
.636 .78 .324 r
.58882 .32983 .60212 .3038 .62885 .26312 .61589 .27642 Metetra
.763 .919 .548 r
.61589 .27642 .62885 .26312 .65586 .23574 .64293 .2353 Metetra
.848 .995 .833 r
.64293 .2353 .65586 .23574 .68328 .21716 .67032 .20755 Metetra
.787 .903 .923 r
.67032 .20755 .68328 .21716 .71111 .20318 .69816 .18865 Metetra
.723 .819 .924 r
.69816 .18865 .71111 .20318 .73929 .19134 .72644 .17437 Metetra
.69 .776 .917 r
.72644 .17437 .73929 .19134 .76781 .18041 .7551 .16223 Metetra
.675 .757 .912 r
.7551 .16223 .76781 .18041 .79663 .16982 .7841 .151 Metetra
.646 .762 .934 r
.1163 .57721 .13432 .59183 .15911 .58394 .14122 .56917 Metetra
.646 .762 .934 r
.14122 .56917 .15911 .58394 .18414 .57597 .16639 .56106 Metetra
.646 .762 .934 r
.16639 .56106 .18414 .57597 .20942 .56793 .19181 .55287 Metetra
.646 .762 .934 r
.19181 .55287 .20942 .56793 .23494 .55981 .21748 .5446 Metetra
.646 .762 .934 r
.21748 .5446 .23494 .55981 .26072 .5516 .24341 .53624 Metetra
.646 .763 .934 r
.24341 .53624 .26072 .5516 .28675 .5433 .26959 .5278 Metetra
.646 .763 .934 r
.26959 .5278 .28675 .5433 .31304 .5349 .29605 .51926 Metetra
.646 .763 .934 r
.29605 .51926 .31304 .5349 .33961 .52637 .32277 .51062 Metetra
.647 .764 .934 r
.32277 .51062 .33961 .52637 .36644 .51767 .34976 .50185 Metetra
.649 .765 .934 r
.34976 .50185 .36644 .51767 .39355 .50865 .37704 .49289 Metetra
.652 .769 .935 r
.37704 .49289 .39355 .50865 .42094 .49904 .4046 .48362 Metetra
.66 .779 .937 r
.4046 .48362 .42094 .49904 .44863 .48819 .43245 .47374 Metetra
.677 .799 .94 r
.43245 .47374 .44863 .48819 .47661 .47471 .4606 .46262 Metetra
.714 .844 .944 r
.4606 .46262 .47661 .47471 .50483 .45585 .48904 .44884 Metetra
.78 .925 .931 r
.48904 .44884 .50483 .45585 .53314 .42709 .5177 .42964 Metetra
.836 .999 .819 r
.5177 .42964 .53314 .42709 .56125 .38422 .54643 .40047 Metetra
.754 .913 .537 r
.54643 .40047 .56125 .38422 .58882 .32983 .57488 .35711 Metetra
.636 .78 .324 r
.57488 .35711 .58882 .32983 .61589 .27642 .60273 .30218 Metetra
.636 .78 .324 r
.60273 .30218 .61589 .27642 .64293 .2353 .63007 .24824 Metetra
.763 .919 .548 r
.63007 .24824 .64293 .2353 .67032 .20755 .65743 .20667 Metetra
.848 .995 .833 r
.65743 .20667 .67032 .20755 .69816 .18865 .68519 .17855 Metetra
.787 .903 .923 r
.68519 .17855 .69816 .18865 .72644 .17437 .71347 .15931 Metetra
.723 .819 .924 r
.71347 .15931 .72644 .17437 .7551 .16223 .74222 .14471 Metetra
.69 .776 .917 r
.74222 .14471 .7551 .16223 .7841 .151 .77136 .13226 Metetra
.646 .762 .934 r
.09795 .56232 .1163 .57721 .14122 .56917 .12301 .55414 Metetra
.646 .762 .934 r
.12301 .55414 .14122 .56917 .16639 .56106 .14832 .54588 Metetra
.646 .762 .934 r
.14832 .54588 .16639 .56106 .19181 .55287 .17388 .53754 Metetra
.646 .762 .934 r
.17388 .53754 .19181 .55287 .21748 .5446 .1997 .52911 Metetra
.646 .762 .934 r
.1997 .52911 .21748 .5446 .24341 .53624 .22577 .5206 Metetra
.646 .762 .934 r
.22577 .5206 .24341 .53624 .26959 .5278 .25212 .512 Metetra
.646 .763 .934 r
.25212 .512 .26959 .5278 .29605 .51926 .27873 .50331 Metetra
.646 .763 .934 r
.27873 .50331 .29605 .51926 .32277 .51062 .30561 .49453 Metetra
.646 .763 .934 r
.30561 .49453 .32277 .51062 .34976 .50185 .33277 .48563 Metetra
.647 .764 .934 r
.33277 .48563 .34976 .50185 .37704 .49289 .36022 .4766 Metetra
.649 .765 .934 r
.36022 .4766 .37704 .49289 .4046 .48362 .38795 .46738 Metetra
.652 .769 .935 r
.38795 .46738 .4046 .48362 .43245 .47374 .41597 .45784 Metetra
.66 .779 .937 r
.41597 .45784 .43245 .47374 .4606 .46262 .4443 .44769 Metetra
.677 .799 .94 r
.4443 .44769 .4606 .46262 .48904 .44884 .47293 .43628 Metetra
.714 .844 .944 r
.47293 .43628 .48904 .44884 .5177 .42964 .50184 .42219 Metetra
.78 .925 .931 r
.50184 .42219 .5177 .42964 .54643 .40047 .53097 .40264 Metetra
.836 .999 .819 r
.53097 .40264 .54643 .40047 .57488 .35711 .56011 .37306 Metetra
.754 .913 .537 r
.56011 .37306 .57488 .35711 .60273 .30218 .58892 .3292 Metetra
.636 .78 .324 r
.58892 .3292 .60273 .30218 .63007 .24824 .61705 .2737 Metetra
.636 .78 .324 r
.61705 .2737 .63007 .24824 .65743 .20667 .64467 .21923 Metetra
.763 .919 .548 r
.64467 .21923 .65743 .20667 .68519 .17855 .67234 .17721 Metetra
.848 .995 .833 r
.67234 .17721 .68519 .17855 .71347 .15931 .7005 .1487 Metetra
.787 .903 .923 r
.7005 .1487 .71347 .15931 .74222 .14471 .72923 .12911 Metetra
.723 .819 .924 r
.72923 .12911 .74222 .14471 .77136 .13226 .75846 .11419 Metetra
.646 .762 .934 r
.07926 .54716 .09795 .56232 .12301 .55414 .10446 .53883 Metetra
.646 .762 .934 r
.10446 .53883 .12301 .55414 .14832 .54588 .12991 .53042 Metetra
.646 .762 .934 r
.12991 .53042 .14832 .54588 .17388 .53754 .15561 .52192 Metetra
.646 .762 .934 r
.15561 .52192 .17388 .53754 .1997 .52911 .18158 .51334 Metetra
.646 .762 .934 r
.18158 .51334 .1997 .52911 .22577 .5206 .20781 .50467 Metetra
.646 .762 .934 r
.20781 .50467 .22577 .5206 .25212 .512 .23431 .49591 Metetra
.646 .762 .934 r
.23431 .49591 .25212 .512 .27873 .50331 .26108 .48705 Metetra
.646 .763 .934 r
.26108 .48705 .27873 .50331 .30561 .49453 .28813 .47811 Metetra
.646 .763 .934 r
.28813 .47811 .30561 .49453 .33277 .48563 .31546 .46906 Metetra
.646 .763 .934 r
.31546 .46906 .33277 .48563 .36022 .4766 .34308 .4599 Metetra
.647 .764 .934 r
.34308 .4599 .36022 .4766 .38795 .46738 .37098 .4506 Metetra
.649 .765 .934 r
.37098 .4506 .38795 .46738 .41597 .45784 .39919 .44111 Metetra
.652 .769 .935 r
.39919 .44111 .41597 .45784 .4443 .44769 .42769 .43129 Metetra
.66 .779 .937 r
.42769 .43129 .4443 .44769 .47293 .43628 .45651 .42085 Metetra
.677 .799 .94 r
.45651 .42085 .47293 .43628 .50184 .42219 .48563 .40913 Metetra
.714 .844 .944 r
.48563 .40913 .50184 .42219 .53097 .40264 .51503 .39473 Metetra
.78 .925 .931 r
.51503 .39473 .53097 .40264 .56011 .37306 .54464 .37481 Metetra
.836 .999 .819 r
.54464 .37481 .56011 .37306 .58892 .3292 .57422 .3448 Metetra
.754 .913 .537 r
.57422 .3448 .58892 .3292 .61705 .2737 .60338 .30043 Metetra
.636 .78 .324 r
.60338 .30043 .61705 .2737 .64467 .21923 .6318 .24437 Metetra
.636 .78 .324 r
.6318 .24437 .64467 .21923 .67234 .17721 .6597 .18935 Metetra
.763 .919 .548 r
.6597 .18935 .67234 .17721 .7005 .1487 .68771 .14687 Metetra
.848 .995 .833 r
.68771 .14687 .7005 .1487 .72923 .12911 .71627 .11796 Metetra
.787 .903 .923 r
.71627 .11796 .72923 .12911 .75846 .11419 .74546 .09802 Metetra
.646 .762 .934 r
.06022 .53172 .07926 .54716 .10446 .53883 .08556 .52323 Metetra
.646 .762 .934 r
.08556 .52323 .10446 .53883 .12991 .53042 .11115 .51466 Metetra
.646 .762 .934 r
.11115 .51466 .12991 .53042 .15561 .52192 .137 .50601 Metetra
.646 .762 .934 r
.137 .50601 .15561 .52192 .18158 .51334 .16312 .49726 Metetra
.646 .762 .934 r
.16312 .49726 .18158 .51334 .20781 .50467 .18951 .48843 Metetra
.646 .762 .934 r
.18951 .48843 .20781 .50467 .23431 .49591 .21616 .4795 Metetra
.646 .762 .934 r
.21616 .4795 .23431 .49591 .26108 .48705 .2431 .47048 Metetra
.646 .762 .934 r
.2431 .47048 .26108 .48705 .28813 .47811 .27031 .46136 Metetra
.646 .763 .934 r
.27031 .46136 .28813 .47811 .31546 .46906 .29781 .45215 Metetra
.646 .763 .934 r
.29781 .45215 .31546 .46906 .34308 .4599 .3256 .44283 Metetra
.646 .763 .934 r
.3256 .44283 .34308 .4599 .37098 .4506 .35369 .43339 Metetra
.647 .764 .934 r
.35369 .43339 .37098 .4506 .39919 .44111 .38208 .42381 Metetra
.649 .765 .934 r
.38208 .42381 .39919 .44111 .42769 .43129 .41077 .41403 Metetra
.652 .769 .935 r
.41077 .41403 .42769 .43129 .45651 .42085 .43977 .40392 Metetra
.66 .779 .937 r
.43977 .40392 .45651 .42085 .48563 .40913 .46909 .39318 Metetra
.677 .799 .94 r
.46909 .39318 .48563 .40913 .51503 .39473 .49873 .38115 Metetra
.714 .844 .944 r
.49873 .38115 .51503 .39473 .54464 .37481 .52864 .36641 Metetra
.78 .925 .931 r
.52864 .36641 .54464 .37481 .57422 .3448 .55873 .34612 Metetra
.836 .999 .819 r
.55873 .34612 .57422 .3448 .60338 .30043 .58876 .31567 Metetra
.754 .913 .537 r
.58876 .31567 .60338 .30043 .6318 .24437 .6183 .27077 Metetra
.636 .78 .324 r
.6183 .27077 .6318 .24437 .6597 .18935 .64701 .21413 Metetra
.636 .78 .324 r
.64701 .21413 .6597 .18935 .68771 .14687 .67518 .15857 Metetra
.763 .919 .548 r
.67518 .15857 .68771 .14687 .71627 .11796 .70353 .11561 Metetra
.848 .995 .833 r
.70353 .11561 .71627 .11796 .74546 .09802 .73251 .0863 Metetra
.646 .762 .934 r
.04083 .51599 .06022 .53172 .08556 .52323 .06631 .50734 Metetra
.646 .762 .934 r
.06631 .50734 .08556 .52323 .11115 .51466 .09205 .49861 Metetra
.646 .762 .934 r
.09205 .49861 .11115 .51466 .137 .50601 .11804 .48979 Metetra
.646 .762 .934 r
.11804 .48979 .137 .50601 .16312 .49726 .14431 .48088 Metetra
.646 .762 .934 r
.14431 .48088 .16312 .49726 .18951 .48843 .17085 .47187 Metetra
.646 .762 .934 r
.17085 .47187 .18951 .48843 .21616 .4795 .19767 .46278 Metetra
.646 .762 .934 r
.19767 .46278 .21616 .4795 .2431 .47048 .22477 .45358 Metetra
.646 .762 .934 r
.22477 .45358 .2431 .47048 .27031 .46136 .25215 .44429 Metetra
.646 .762 .934 r
.25215 .44429 .27031 .46136 .29781 .45215 .27982 .4349 Metetra
.646 .763 .934 r
.27982 .4349 .29781 .45215 .3256 .44283 .30779 .4254 Metetra
.646 .763 .934 r
.30779 .4254 .3256 .44283 .35369 .43339 .33606 .4158 Metetra
.646 .763 .934 r
.33606 .4158 .35369 .43339 .38208 .42381 .36463 .40607 Metetra
.647 .764 .934 r
.36463 .40607 .38208 .42381 .41077 .41403 .39351 .3962 Metetra
.649 .765 .934 r
.39351 .3962 .41077 .41403 .43977 .40392 .42271 .38612 Metetra
.652 .769 .935 r
.42271 .38612 .43977 .40392 .46909 .39318 .45223 .3757 Metetra
.66 .779 .937 r
.45223 .3757 .46909 .39318 .49873 .38115 .48207 .36464 Metetra
.677 .799 .94 r
.48207 .36464 .49873 .38115 .52864 .36641 .51223 .35229 Metetra
.714 .844 .944 r
.51223 .35229 .52864 .36641 .55873 .34612 .54267 .3372 Metetra
.78 .925 .931 r
.54267 .3372 .55873 .34612 .58876 .31567 .57328 .31652 Metetra
.836 .999 .819 r
.57328 .31652 .58876 .31567 .6183 .27077 .60377 .28562 Metetra
.754 .913 .537 r
.60377 .28562 .6183 .27077 .64701 .21413 .63368 .24018 Metetra
.636 .78 .324 r
.63368 .24018 .64701 .21413 .67518 .15857 .66269 .18296 Metetra
.636 .78 .324 r
.66269 .18296 .67518 .15857 .70353 .11561 .69115 .12684 Metetra
.763 .919 .548 r
.69115 .12684 .70353 .11561 .73251 .0863 .71984 .0834 Metetra
.646 .762 .934 r
.02108 .49996 .04083 .51599 .06631 .50734 .0467 .49115 Metetra
.646 .762 .934 r
.0467 .49115 .06631 .50734 .09205 .49861 .07257 .48226 Metetra
.646 .762 .934 r
.07257 .48226 .09205 .49861 .11804 .48979 .09872 .47327 Metetra
.646 .762 .934 r
.09872 .47327 .11804 .48979 .14431 .48088 .12514 .46418 Metetra
.646 .762 .934 r
.12514 .46418 .14431 .48088 .17085 .47187 .15184 .455 Metetra
.646 .762 .934 r
.15184 .455 .17085 .47187 .19767 .46278 .17881 .44573 Metetra
.646 .762 .934 r
.17881 .44573 .19767 .46278 .22477 .45358 .20608 .43635 Metetra
.646 .762 .934 r
.20608 .43635 .22477 .45358 .25215 .44429 .23363 .42688 Metetra
.646 .762 .934 r
.23363 .42688 .25215 .44429 .27982 .4349 .26148 .41731 Metetra
.646 .762 .934 r
.26148 .41731 .27982 .4349 .30779 .4254 .28962 .40763 Metetra
.646 .763 .934 r
.28962 .40763 .30779 .4254 .33606 .4158 .31807 .39784 Metetra
.646 .763 .934 r
.31807 .39784 .33606 .4158 .36463 .40607 .34683 .38793 Metetra
.646 .763 .934 r
.34683 .38793 .36463 .40607 .39351 .3962 .37591 .3779 Metetra
.647 .764 .934 r
.37591 .3779 .39351 .3962 .42271 .38612 .4053 .36772 Metetra
.649 .765 .934 r
.4053 .36772 .42271 .38612 .45223 .3757 .43503 .35733 Metetra
.652 .769 .935 r
.43503 .35733 .45223 .3757 .48207 .36464 .46508 .34658 Metetra
.66 .779 .937 r
.46508 .34658 .48207 .36464 .51223 .35229 .49546 .3352 Metetra
.677 .799 .94 r
.49546 .3352 .51223 .35229 .54267 .3372 .52617 .3225 Metetra
.714 .844 .944 r
.52617 .3225 .54267 .3372 .57328 .31652 .55716 .30705 Metetra
.78 .925 .931 r
.55716 .30705 .57328 .31652 .60377 .28562 .58829 .28597 Metetra
.836 .999 .819 r
.58829 .28597 .60377 .28562 .63368 .24018 .61925 .2546 Metetra
.754 .913 .537 r
.61925 .2546 .63368 .24018 .66269 .18296 .64955 .20861 Metetra
.636 .78 .324 r
.64955 .20861 .66269 .18296 .69115 .12684 .67887 .1508 Metetra
.636 .78 .324 r
.67887 .1508 .69115 .12684 .71984 .0834 .70761 .09411 Metetra
.646 .762 .934 r
.00095 .48363 .02108 .49996 .0467 .49115 .02671 .47466 Metetra
.646 .762 .934 r
.02671 .47466 .0467 .49115 .07257 .48226 .05273 .46559 Metetra
.646 .762 .934 r
.05273 .46559 .07257 .48226 .09872 .47327 .07903 .45642 Metetra
.646 .762 .934 r
.07903 .45642 .09872 .47327 .12514 .46418 .1056 .44716 Metetra
.646 .762 .934 r
.1056 .44716 .12514 .46418 .15184 .455 .13245 .4378 Metetra
.646 .762 .934 r
.13245 .4378 .15184 .455 .17881 .44573 .15959 .42835 Metetra
.646 .762 .934 r
.15959 .42835 .17881 .44573 .20608 .43635 .18702 .41879 Metetra
.646 .762 .934 r
.18702 .41879 .20608 .43635 .23363 .42688 .21474 .40913 Metetra
.646 .762 .934 r
.21474 .40913 .23363 .42688 .26148 .41731 .24276 .39936 Metetra
.646 .762 .934 r
.24276 .39936 .26148 .41731 .28962 .40763 .27109 .38949 Metetra
.646 .762 .934 r
.27109 .38949 .28962 .40763 .31807 .39784 .29973 .3795 Metetra
.646 .763 .934 r
.29973 .3795 .31807 .39784 .34683 .38793 .32868 .36941 Metetra
.646 .763 .934 r
.32868 .36941 .34683 .38793 .37591 .3779 .35795 .35919 Metetra
.646 .763 .934 r
.35795 .35919 .37591 .3779 .4053 .36772 .38754 .34885 Metetra
.647 .764 .934 r
.38754 .34885 .4053 .36772 .43503 .35733 .41747 .33834 Metetra
.649 .765 .934 r
.41747 .33834 .43503 .35733 .46508 .34658 .44774 .32761 Metetra
.652 .769 .935 r
.44774 .32761 .46508 .34658 .49546 .3352 .47834 .31654 Metetra
.66 .779 .937 r
.47834 .31654 .49546 .3352 .52617 .3225 .50929 .30481 Metetra
.677 .799 .94 r
.50929 .30481 .52617 .3225 .55716 .30705 .54056 .29175 Metetra
.714 .844 .944 r
.54056 .29175 .55716 .30705 .58829 .28597 .57211 .27592 Metetra
.78 .925 .931 r
.57211 .27592 .58829 .28597 .61925 .2546 .60379 .25442 Metetra
.836 .999 .819 r
.60379 .25442 .61925 .2546 .64955 .20861 .63524 .22257 Metetra
.754 .913 .537 r
.63524 .22257 .64955 .20861 .67887 .1508 .66594 .17602 Metetra
.636 .78 .324 r
.66594 .17602 .67887 .1508 .70761 .09411 .69556 .11761 Metetra
0 g
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.1552 .20892 m
.15981 .21359 L
s
[(0.2)] .14598 .19958 .98733 1 Mshowa
.27893 .16047 m
.28333 .16533 L
s
[(0.4)] .27013 .15073 .90393 1 Mshowa
.40878 .10962 m
.41294 .11469 L
s
[(0.6)] .40046 .09948 .82054 1 Mshowa
.54521 .0562 m
.54911 .06148 L
s
[(0.8)] .53743 .04564 .73714 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(1)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.18558 .19702 m
.18832 .19985 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.3108 .14799 m
.31341 .15094 L
s
.34306 .13535 m
.34564 .13834 L
s
.37572 .12257 m
.37826 .12558 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.51046 .06981 m
.51284 .07294 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[52]:=",
  ImageSize->{288, 233.563},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jB000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00aP3oool5
000005D0oooo00380?ooo`030000003oool0oooo05D0oooo00380?ooo`030000003oool0oooo05D0
oooo00380?ooo`030000003oool0oooo05D0oooo00380?ooo`030000003oool0oooo05D0oooo0037
0?ooo`800000E`3oool00=40oooo0P00001=0?ooo`00d03oool010000000oooo0?ooo`00001<0?oo
o`00d03oool010000000oooo0?ooo`00001<0?ooo`00d03oool010000000oooo0?ooo`00001<0?oo
o`00d03oool010000000oooo0?ooo`00001<0?ooo`00b`3oool2000000@0oooo0P00001=0?ooo`00
b@3oool500000580oooo00360?ooo`<000000`3oool200000580oooo00340?ooo`8000001P3oool0
0`000000oooo0000001A0?ooo`00U`3oool2000000@0oooo0P0000040?ooo`800000703oool30000
00P0oooo00@000000?ooo`3oool00000D03oool009H0oooo00@000000?ooo`3oool00000203oool0
10000000oooo0?ooo`00000I0?ooo`8000000P3oool00`000000oooo0?ooo`060?ooo`040000003o
ool0oooo00000500oooo002F0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
00005P3oool3000000d0oooo00D000000?ooo`3oool0oooo0000001?0?ooo`00UP3oool010000000
oooo0?ooo`0000090?ooo`800000503oool300000100oooo00<000000?ooo`3oool00`00001>0?oo
o`00UP3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000140oooo0`00000B
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0C03oool009L0oooo0P00000:0?oo
o`8000003`3oool3000000<0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo04/0oooo002b0?ooo`8000006@3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo04X0oooo002_0?ooo`<000006`3oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo04X0oooo002]0?ooo`80000000<0oooo0000003oool06`3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo04T0oooo002Z0?ooo`<00000803oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo04T0oooo002X0?ooo`8000008`3oool00`000000oooo0?ooo`040?oo
o`800000BP3oool00:D0oooo0`00000U0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3o
ool0A`3oool00:<0oooo0P0000020?ooo`030000003oool0oooo02<0oooo00<000000?ooo`3oool0
1P3oool00`000000oooo0?ooo`170?ooo`00X03oool3000002X0oooo00<000000?ooo`3oool01`3o
ool00`000000oooo0?ooo`160?ooo`00LP3oool2000000@0oooo0P0000030?ooo`<000007@3oool3
000002d0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`150?ooo`00L@3oool01000
0000oooo0?ooo`0000080?ooo`040000003oool0oooo000001X0oooo1000000^0?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool0A@3oool00740oooo00@000000?ooo`3oool00000203o
ool010000000oooo0?ooo`00000G0?ooo`<00000<P3oool00`000000oooo0?ooo`090?ooo`030000
003oool0oooo04@0oooo001a0?ooo`040000003oool0oooo000000P0oooo0`00000F0?ooo`800000
=@3oool00`000000oooo0?ooo`080?ooo`030000003oool0000000L0oooo0P0000040?ooo`800000
0`3oool4000002l0oooo001a0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool0
4P3oool3000003L0oooo00<000000?ooo`3oool0103oool2000000@0oooo00<000000?ooo`3oool0
103oool010000000oooo0?ooo`0000080?ooo`030000003oool0oooo0300oooo001b0?ooo`800000
2P3oool300000100oooo0P0000020?ooo`030000003oool0oooo03D0oooo00<000000?ooo`3oool0
0`3oool00`000000`n^<0<?[S002000000<0oooo00<000000?ooo`3oool00`3oool010000000oooo
0?ooo`0000090?ooo`030000003oool0oooo02l0oooo002>0?ooo`<00000?03oool00`000000oooo
0?ooo`020?ooo`8000000P33jh`00`000000fOkE000000030?ooo`030000003oool0oooo0080oooo
00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`0^0?ooo`00S03oool200000400oooo
00<000000?ooo`3oool00P0000020<?[S0040000003Io]D0fOkE0=Wne@80000000<0oooo0000003o
ool00`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000002l0oooo00290?oo
o`@00000@@3oool010000000oooo0000000000020<?[S0030000003Io]D0fOkE00@0fOkE00<00000
0?ooo`000000103oool2000000X0oooo0P00000`0?ooo`00Q`3oool2000004D0oooo00H000000?oo
o`00000000000<?[S00000050=Wne@80000000@0bNO/0000000000000000@@3oool008@0oooo0`00
00170?ooo`<0000000<0`n^<0000003Io]D0103Io]D01`000000bNO/0<WWk039in`000000?ooo`00
0000@@3oool00880oooo0P00001:0?ooo`@00000103Io]D2000000D0bNO/00<000000?ooo`000000
@03oool007l0oooo0`0000020?ooo`030000003oool0oooo04L0oooo100000020=Wne@8000002039
in`00`000000oooo0000000o0?ooo`00O@3oool2000004l0oooo0P0000020<?[S08000002039in`2
000000030;WAk0000000000003l0oooo001?0?ooo`800000103oool2000000D0oooo0`00000K0?oo
o`<00000D03oool300000080`n^<00<000000=Wne@0000001@39in`2000000@0^M7/0P00000n0?oo
o`00CP3oool010000000oooo0?ooo`00000;0?ooo`030000003oool0oooo01P0oooo0P0000000`3o
ool000000?ooo`1@0?ooo`8000000P33jh`010000000fOkE0=Wne@3Io]D2000000030<WWk0000000
000000H0^M7/0`00000m0?ooo`00CP3oool010000000oooo0?ooo`0000080?ooo`D000005P3oool3
000005@0oooo0`0000000`33jh`000000=Wne@050=Wne@030000002idN`0^M7/00L0^M7/0P00000m
0?ooo`00CP3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000001@0oooo0`00
001G0?ooo`@000001@3Io]D2000000030<WWk0000000^M7/00L0^M7/0`00000l0?ooo`00CP3oool0
10000000oooo0?ooo`0000090?ooo`030000003oool000000180oooo0`00001H0?ooo`80000000<0
`n^<000000000000103Io]D01@000000bNO/0<WWk039in`0000000H0^M7/0P0000020;36jP030000
003oool0oooo03X0oooo001?0?ooo`8000002`3oool2000000l0oooo0`00001K0?ooo`0400000033
jh`0`n^<000000<0fOkE0P0000050<WWk0030000002idN`0^M7/00<00000102`a^X2000003/0oooo
001[0?ooo`800000G@3oool2000000040<?[S000000000000=Wne@8000002039in`2000000P0/<KZ
0P00000j0?ooo`00J03oool3000005l0oooo00<000000<?[S00000000`0000080<WWk080000000<0
^M7/0000002`a^X01`2`a^X2000003X0oooo001V0?ooo`8000000P3oool00`000000oooo0?ooo`1L
0?ooo`0300000033jh`000000080`n^<00<000000=Wne@0000001@39in`2000000<0^M7/00<00000
0;36jP2`a^X01P2`a^X3000003T0oooo001S0?ooo`<00000HP3oool4000000060<?[S0000000fOkE
0000003Io]D000000P39in`2000000H0^M7/00<000000;36jP2`a^X00`2`a^X200000080[<7Y0P00
000h0?ooo`00H@3oool2000006D0oooo00D000000:;7DP000000`n^<000000020=Wne@030000003I
o]D0fOkE00800000202idN`00`000000/<KZ0;36jP03000000@0[<7Y0P00000h0?ooo`00GP3oool3
000000030?ooo`000000oooo06<0oooo0`0000000`33jh`000000=Wne@020=Wne@<0000000<0bNO/
0000002idN`01`2idN`3000000P0[<7Y0P00000g0?ooo`00;P3oool2000000@0oooo0P0000030?oo
o`@000007`3oool2000006T0oooo0P0000020<?[S0050000003Io]D0fOkE0=Wne@0000001039in`0
0`000000^M7/0;WAk0030;WAk08000000P2`a^X00`000000[<7Y0:c1j@060:c1j@030000003oool0
000003H0oooo000]0?ooo`040000003oool0oooo000000P0oooo00<000000?ooo`3oool07@3oool3
000006X0oooo0`0000000`33jh`000000=Wne@020=Wne@8000001@39in`00`000000^M7/0;WAk002
000000D0/<KZ00<000000:c1j@2/`NT01@2/`NT3000003H0oooo000]0?ooo`040000003oool0oooo
000000T0oooo00<000000?ooo`3oool06P3oool3000006`0oooo0P0000001033jh`000000=Wne@3I
o]D2000000030<WWk0000000bNO/00@0bNO/0`0000080;36jP030000002/`NT0[<7Y00@000000P2[
_nP2000000L0oooo0P0000040?ooo`8000001@3oool3000001h0oooo000]0?ooo`040000003oool0
oooo000000X0oooo00<000000?ooo`3oool05P3oool3000006h0oooo0P0000000`33jh`000000000
0002000000<0bNO/00@000000<WWk039in`0bNO/0P0000020;WAk0030000002`a^X0/<KZ00H0/<KZ
0`0000060:^oj0030000003oool0000000D0oooo00@000000?ooo`3oool000002`3oool00`000000
oooo0?ooo`0M0?ooo`00;@3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo0000
01@0oooo0`00001a0?ooo`0300000033jh`000000080`n^<00D000000=Wne@00000000000<WWk004
000000@0^M7/00<000000;36jP2`a^X0102`a^X200000080[<7Y00<000000:^oj02[_nP01@2[_nP2
000000D0oooo00@000000?ooo`3oool00000203oool5000001h0oooo000^0?ooo`8000002P3oool2
000001<0oooo0`00001b0?ooo`<000000P33jh`00`000000fOkE0=Wne@020=Wne@800000202idN`0
0`000000/<KZ0;36jP03000000@0[<7Y00<000000:^oj02[_nP01@2[_nP3000000@0oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`00000O0?ooo`00C03oool3000007@0oooo0P00
0000102Rae8000000<?[S00000030=Wne@80000000<0bNO/0000002idN`01`2idN`3000000P0[<7Y
00<000000:^oj02[_nP00P2[_nP200000080Z[kX0P0000030?ooo`040000003oool0oooo000000T0
oooo00<000000?ooo`0000007`3oool004X0oooo0P00001g0?ooo`<0000000<0`n^<0000003Io]D0
0P3Io]D010000000bNO/0<WWk039in`2000000D0^M7/0P0000020;36jP030000002/`NT0[<7Y00H0
[<7Y00<000000:^oj02[_nP00P0000040:Znj0800000103oool2000000/0oooo0P00000O0?ooo`00
A`3oool3000007P0oooo0`0000000`33jh`000000=Wne@020=Wne@0300000039in`0bNO/0080bNO/
00<000000<WWk00000000P2idN`2000000@0/<KZ00<000000:c1j@2/`NT01P2/`NT3000000L0Z[kX
0P00000a0?ooo`00A@3oool2000007X0oooo00D000000<7YR@000000`n^<000000020=Wne@800000
1@39in`00`000000bNO/0<WWk002000000L0/<KZ00<000000:c1j@2/`NT00`2/`NT200000080ZkoX
00<000000:Znj02Z_^P01@2Z_^P00`000000oooo0000000`0?ooo`00@P3oool3000007`0oooo0P00
00000`33jh`0000000000002000000L0bNO/0`0000000`2idN`000000;36jP060;36jP030000002/
`NT0[<7Y00<00000102[_nP00`000000Z[kX0:Znj0050:Znj0<00000<03oool00400oooo0P00001n
0?ooo`0400000031jHT0000000000080`n^<00<000000=Wne@0000001@39in`2000000<0^M7/00<0
00000;36jP2`a^X01@2`a^X3000000P0ZkoX00<000000:Znj02Z_^P00P2Z_^P200000080ZKkX0P00
000_0?ooo`00?@3oool300000080oooo00<000000?ooo`3oool0NP3oool00`000000`NV900000002
000000030<?[S0000000fOkE0080fOkE0P0000001@39in`000000000002idN`0000000@0^M7/00<0
00000;36jP2`a^X00P2`a^X200000080[<7Y00<000000:^oj02[_nP01P2[_nP00`000000Z[kX0:Zn
j002000000D0ZKkX00<000000?ooo`3oool0;@3oool001/0oooo0P00000N0?ooo`800000PP3oool4
000000030<?[S0000000fOkE00D0fOkE00<000000;WAk02idN`00P2idN`01@000000^M7/0;WAk02i
dN`000000080/<KZ0P0000050:c1j@030000002[_nP0ZkoX00D0ZkoX0`0000070:Vnj0800000;P3o
ool001X0oooo00@000000?ooo`3oool000006P3oool3000008<0oooo00H000000=Kod@0000000000
0<?[S00000050=Wne@80000000<0bNO/0000002idN`00P2idN`010000000^M7/0;WAk02idN`30000
00P0[<7Y00<000000:^oj02[_nP0100000020:Znj0030000002Y_^P0ZKkX00H0ZKkX0P00000]0?oo
o`006P3oool010000000oooo0?ooo`00000H0?ooo`80000000<0oooo0000003oool0PP3oool00`00
0000e_oA00000002000000030<?[S0000000fOkE00<0fOkE00D000000<WWk039in`0bNO/00000003
0;WAk0040000002idN`0000000000080/<KZ00<000000:c1j@2/`NT01P2/`NT3000000H0Z[kX00<0
00000:Vnj02Y_^P01P2Y_^P2000002d0oooo000J0?ooo`040000003oool0oooo000001D0oooo0`00
00260?ooo`030000003Fom40e_oA0080000000<0`n^<0000003Io]D00P3Io]D2000000D0bNO/00<0
00000;WAk02idN`00P0000040;36jP030000002/`NT0[<7Y00@0[<7Y0P0000020:^oj0030000002Z
_^P0Z[kX00D0Z[kX00<000000:Vnj02Y_^P00`2Y_^P200000080ZKgX00<000000?ooo`3oool0:P3o
ool001X0oooo00@000000?ooo`3oool000004P3oool3000008P0oooo00<000000=Kod@3Fom401000
00020=Wne@8000001`39in`4000000H0/<KZ00<000000:c1j@2/`NT00`0000040:^oj0030000002Z
_^P0Z[kX00D0Z[kX00<000000:Vnj02Y_^P00`0000040:Vmj0800000:`3oool001/0oooo0P00000A
0?ooo`8000000P3oool00`000000oooo0?ooo`250?ooo`030000003Fom40e_oA00L000001`39in`2
00000080^M7/0P0000060;36jP<00000202[_nP00`000000Z[kX0:Znj0040:Znj0<00000202Y_NP0
0`000000oooo0?ooo`0Y0?ooo`00:`3oool3000008`0oooo00<000000=Kod@3Fom400`0000020<?[
S0040000003Io]D00000000000<0bNO/0P0000040;WAk0800000102`a^X200000080[<7Y00<00000
0:^oj02[_nP01P2[_nP00`000000Z[kX0:Znj00400000080ZKkX00<000000:Vmj02Y_NP01P2Y_NP2
000002X0oooo000Y0?ooo`800000SP3oool00`000000e_oA0=Kod@04000000030<?[S0000000fOkE
00<0fOkE0`0000060;WAk08000000P2`a^X2000000D0[<7Y00<000000:^oj02[_nP01@2[_nP30000
00H0ZKkX00<000000:Vmj02Y_NP01P2Y_NP3000002T0oooo000V0?ooo`<0000000<0oooo0000003o
ool0S03oool00`000000e_oA0=Kod@04000000030<?[S0000000fOkE00<0fOkE0P0000000`39in`0
00000;WAk0020;WAk0<0000000@0^M7/00000000000000001`2/`NT00`000000ZkoX0:^oj0030:^o
j08000000P2Z_^P00`000000ZKkX0:Vnj0050:Vnj0040000002Y_NP0ZKgX0:Vmj0@000000P2Y_NP0
0`000000oooo0?ooo`0W0?ooo`00903oool200000940oooo00@000000=Kod@3Fom40e_oA0`000000
0`33jh`000000=Wne@030=Wne@0400000039in`0bNO/0<WWk0@000000P2idN`2000000030;36jP00
0000000000P0[<7Y00<000000:^oj02[_nP00P0000040:Znj0030000002Y_^P0ZKkX00D0ZKkX1000
00060:Vmj0800000:03oool00280oooo0P00002B0?ooo`L0000000<0`n^<0000003Io]D00`3Io]D0
0`000000bNO/0<WWk00400000080^M7/0P0000040;36jP8000001`2/`NT3000000L0Z[kX00<00000
0:Vnj02Y_^P00P2Y_^P200000080ZKgX00<000000:Vmj02Y_NP01P2Y_NP2000002L0oooo000R0?oo
o`800000T03oool2000000<0anc^00<000000=Kod@0000000P0000030=Wne@80000000<0bNO/0000
000000000`39in`3000000L0/<KZ0P0000020:c1j@@000000P2[_nP00`000000Z[kX0:Znj0050:Zn
j0030000002Y_^P0ZKkX00800000102Y_NP00`000000ZKgX0:Vmj0060:Vmj0800000203oool20000
00@0oooo0P0000030?ooo`<000004@3oool00280oooo00<000000?ooo`000000S@3oool2000000@0
anc^00<000000=Kod@0000002@0000030<WWk08000000P2idN`00`000000/<KZ0;36jP060;36jP<0
00001P2[_nP00`000000Z[kX0:Znj0050:Znj0<000001`2Y_NP00`000000ZKgX0:Vmj0030:Vmj080
00000P2Y_NP00`000000oooo0?ooo`040?ooo`040000003oool0oooo000000P0oooo00@000000?oo
o`3oool00000403oool001X0oooo0P0000060?ooo`040000003oool0oooo000008X0oooo0P000005
0<O/kP030000003Fom400000008000000P33jh`500000080bNO/0P0000040;WAk0030000002`a^X0
/<KZ00@0/<KZ0`0000000`2/`NT000000:^oj0060:^oj0030000002Z_^P0Z[kX00@000000P2Y_^P0
0`000000ZKgX0:Vmj0050:Vmj0030000002Y_NP0ZKgX00<00000102Y_NP2000000D0oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`00000@0?ooo`006@3oool010000000oooo0?oo
o`0000040?ooo`8000000`3oool00`000000oooo0?ooo`250?ooo`8000001P37k>h00`000000e_oA
0=Kod@06000000@0fOkE0P0000070;WAk0030000002`a^X0/<KZ00<000000`2/`NT2000000L0ZkoX
0`0000060:Vnj0030000002Y_NP0ZKgX00D0ZKgX0`0000080:Vmj0030000003oool0oooo00<0oooo
00@000000?ooo`3oool00000203oool300000140oooo000I0?ooo`040000003oool0oooo000000@0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`230?ooo`0300000037k>h0anc^00D0
anc^00<000000=Kod@3Fom400`0000020<?[S0040000003Io]D0fOkE0=Wne@80000000<0bNO/0000
002idN`01P2idN`3000000L0[<7Y0P0000040:^oj08000000P2Z_^P00`000000ZKkX0:Vnj0050:Vn
j0030000002Y_NP0ZKgX00@000000P2Y_NP00`000000ZKgX0:Vmj0060:Vmj0800000103oool01000
0000oooo0?ooo`0000090?ooo`030000003oool0oooo0100oooo000I0?ooo`040000003oool0oooo
000000@0oooo0P0000050?ooo`030000003oool0oooo0800oooo0P0000000`2ff?40000000000004
0<O/kPH000000P33jh`01@000000fOkE0=Wne@3Io]D0000000<0bNO/00<000000;WAk02idN`00`2i
dN`200000080/<KZ00<000000:c1j@2/`NT01@2/`NT200000080ZkoX0P0000040:Znj0030000002Y
_^P0ZKkX00D0ZKkX0`0000060:Vmj0030000002Y_NP0ZKgX00H0ZKgX0`0000040?ooo`8000002P3o
ool300000100oooo000I0?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`1m0?ooo`8000001@2ff?4500000080e_oA0`0000000`33jh`000000=Wn
e@020=Wne@0300000039in`0bNO/00<0bNO/00<000000;WAk02idN`00P0000050;36jP030000002/
`NT0[<7Y00D0[<7Y0`0000070:Znj0030000002Y_^P0ZKkX00@000000P2Y_NP00`000000ZKgX0:Vm
j0050:Vmj0040000002Y_NP0ZKgX0:Vmj0@0000000<0ZKgX0000000000008`3oool001X0oooo0P00
00050?ooo`030000003oool0oooo00H0oooo0P00001k0?ooo`8000001P2ff?4:000000040<?[S000
0000fOkE0=Wne@8000001P39in`3000000P0/<KZ00<000000:c1j@2/`NT0100000000`2[_nP00000
000000070:Znj0<000001P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0@000001P2Y_NP200000280oooo
000P0?ooo`8000002P3oool00`000000oooo0?ooo`1f0?ooo`8000001P2ff?42000000030;KHl@00
0000000000<0anc^00<000000=Kod@0000001@0000060<WWk08000000P2idN`00`000000/<KZ0;36
jP060;36jP<000001P2[_nP00`000000Z[kX0:Znj0030:Znj08000000P2Y_^P00`000000ZKgX0:Vm
j0050:Vmj0030000002Y_NP0ZKgX00@000000P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0030000003o
ool000000240oooo000P0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0L`3oool2
000000030:g<l0000000]]Sa0080]]Sa0`0000020;KHl@0300000037k>h0anc^0080anc^00<00000
0=Kod@0000000P0000020<?[S0040000003Io]D0000000000080bNO/0P0000050;WAk0030000002`
a^X0/<KZ00<0/<KZ0P0000020:c1j@030000002[_nP0ZkoX00@0ZkoX0P0000020:Znj0800000102Y
_^P00`000000ZKgX0:Vmj0050:Vmj0<000001P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000008@3o
ool00200oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`1`0?ooo`800000102]c?03
000000<0]]Sa0P0000040<O/kP030000003Fom40000000<0000000<0`n^<0000003Io]D00`3Io]D2
000000P0^M7/00<000000;36jP2`a^X00P0000040:c1j@030000002[_nP0ZkoX00D0ZkoX0`000007
0:Vnj0030000002Y_NP0ZKgX0080ZKgX0P0000020:Vmj0030000002Y_NP0ZKgX00D0ZKgX00<00000
0:Vmj02Y_NP0100000020:Vmj0800000803oool00200oooo0P00000=0?ooo`030000003oool0oooo
06d0oooo0P0000030:g<l0<000000P2]c?0010000000]]Sa0000000000050<O/kP030000003Fom40
e_oA00<0000000<0`n^<0000003Io]D00`3Io]D00`000000bNO/000000080;WAk0<000001`2/`NT0
0`000000ZkoX0:^oj004000000030:Znj0000000000000L0ZKkX00<000000:Vmj02Y_NP00P000004
0:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`0000060:Vmj0800000803oool00200oooo00<000000?oo
o`3oool03@3oool00`000000oooo0?ooo`1Y0?ooo`<000000P2]c?03000000D0[Lc`0P0000060<O/
kP030000003Fom40e_oA00<0000000<0`n^<0000003Io]D00`3Io]D010000000bNO/0<WWk039in`2
00000080^M7/100000020;36jP030000002/`NT0[<7Y00D0[<7Y0`0000060:Znj0030000002Y_^P0
ZKkX00D0ZKkX0`0000070:Vmj0030000002Y_NP0ZKgX0080ZKgX0P0000020:Vmj0030000002Y_NP0
ZKgX00D0ZKgX0P00000O0?ooo`00803oool2000000l0oooo00<000000?ooo`3oool0IP3oool20000
0080Z<O_0`0000060:g<l08000000P2ff?400`000000anc^0<O/kP020<O/kP030000003Fom40e_oA
00D000000`3Io]D2000000H0bNO/0P0000060;36jP030000002/`NT0[<7Y00<0[<7Y0P0000020:^o
j0030000002Z_^P0Z[kX00@0Z[kX0P0000020:Vnj0@000000P2Y_NP00`000000ZKgX0:Vmj0050:Vm
j0030000002Y_NP0ZKgX00800000102Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000007P3oool001l0
oooo0P00000A0?ooo`030000003oool0oooo0680oooo0`0000001P2Xanl00000000000000000Z<O_
000000D0[Lc`0P0000050;KHl@80000000<0anc^0000003Fom400P3Fom44000000<0fOkE00<00000
0<WWk039in`01039in`2000000030;WAk0000000/<KZ00H0/<KZ00<000000:c1j@2/`NT00P000004
0:^oj0030000002Z_^P0Z[kX00D0Z[kX0`0000060:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`000007
0:Vmj0030000002Y_NP0ZKgX0080ZKgX0P0000020:Vmj0030000003oool0oooo01`0oooo000O0?oo
o`030000003oool0oooo0140oooo00<000000?ooo`3oool0GP3oool3000000040:S7k`0000000000
000000D0Z<O_00<000000:g<l02]c?000P0000090;KHl@H0000000<0X/MB0000000000000`000005
0<WWk08000000`2idN`00`000000/<KZ0;36jP050;36jP<000001`2[_nP00`000000Z[kX0:Znj002
0:Znj080000000<0ZKkX0000000000001`2Y_NP00`000000ZKgX0:Vmj00400000080ZKgX00<00000
0:Vmj02Y_NP01@2Y_NP00`000000ZKgX0:Vmj002000000@0ZKgX0P00000M0?ooo`001P3oool20000
00@0oooo0P0000030?ooo`@000000`3oool2000000D0oooo00<000000?ooo`3oool04P3oool00`00
0000oooo0?ooo`1K0?ooo`8000000P2Va>l2000000T0Z<O_0P0000090;KHl@8000000`37k>h40000
00<0`n^<00D000000=Wne@00000000000<WWk002000000H0^M7/00<000000;36jP2`a^X010000002
0:c1j@030000002[_nP0ZkoX00D0ZkoX00<000000:Znj02Z_^P00P0000030:Vnj08000001`2Y_NP3
000000H0ZKgX00<000000:Vmj02Y_NP01@2Y_NP3000000L0ZKgX0P00000L0?ooo`001@3oool01000
0000oooo0?ooo`0000080?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool00000103o
ool2000001@0oooo00<000000?ooo`3oool0E`3oool3000000060:K4k`00000000000000002Va>l0
00001`2Xanl2000000030:g<l0000000]]Sa00L0]]Sa00<000000<O/kP37k>h00P37k>h00`000000
e_oA0000000200000080`n^<00<000000=Wne@3Io]D00P3Io]D00`000000^M7/0;WAk0060;WAk0<0
00001P2/`NT00`000000ZkoX0:^oj0050:^oj0<000001P2Y_^P200000080ZKgX100000020:Vmj003
0000002Y_NP0ZKgX00D0ZKgX00<000000:Vmj02Y_NP0100000020:Vmj0030000002Y_NP0ZKgX00D0
ZKgX0P00000L0?ooo`001@3oool010000000oooo0?ooo`0000090?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo05<0
oooo0`000000102Va>l000000000000000040:K4k`030000002Xanl0Z<O_00<0Z<O_0P0000040:g<
l0800000102ff?42000000@0anc^00<000000=Kod@0000000P0000020<?[S0040000003Io]D0fOkE
0=Wne@80000000<0bNO/0000002idN`0102idN`200000080/<KZ00<000000:c1j@2/`NT01@2/`NT0
0`000000ZkoX0:^oj00400000080Z[kX00<000000:Vnj02Y_^P01@2Y_^P3000000H0ZKgX00<00000
0:Vmj02Y_NP01@2Y_NP3000000H0ZKgX00<000000:Vmj02Y_NP01@2Y_NP3000001/0oooo00050?oo
o`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3oool2000000D0oooo00<00000
0?ooo`3oool05@3oool00`000000oooo0?ooo`1@0?ooo`80000000@0Y/?_0000000000000000202V
a>l00`000000Z<O_0:S7k`02000000P0[Lc`00@000000;KHl@00000000001@37k>h00`000000e_oA
0000000200000080`n^<00<000000=Wne@3Io]D00P0000040<WWk0030000002idN`0^M7/00800000
102`a^X00`000000[<7Y0:c1j@050:c1j@<000001P2Z_^P00`000000ZKkX0:Vnj0030:Vnj0800000
00<0ZKgX0000000000001`2Y_NP00`000000ZKgX0:Vmj00400000080ZKgX00<000000:Vmj02Y_NP0
1@2Y_NP00`000000ZKgX0:Vmj00400000080ZKgX0P00000J0?ooo`001@3oool010000000oooo0?oo
o`0000080?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool00`3oool2000001P0
oooo00<000000?ooo`3oool0C03oool600000080Y/?_00<000000:K4k`2Va>l01P2Va>l3000000/0
[Lc`00<000000<O/kP37k>h01037k>h00`000000e_oA0000000200000080`n^<00@000000=Wne@00
000000001`39in`2000000L0/<KZ00<000000:c1j@2/`NT0100000020:^oj0030000002Z_^P0Z[kX
00D0Z[kX00<000000:Vnj02Y_^P00P0000030:Vmj08000001`2Y_NP3000000H0ZKgX00<000000:Vm
j02Y_NP01@2Y_NP3000000H0ZKgX0P00000J0?ooo`001P3oool2000000X0oooo0P0000040?ooo`<0
00000`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo04P0oooo1P0000060:K3k`03
0000002Va>l0Y/C_00<0Y/C_0P0000020:S7k`030000002]c?00[Lc`00L0[Lc`0P0000000`2ff?40
00000<O/kP030<O/kP030000003Fom40e_oA00<0000000@0`n^<00000000000000001`39in`20000
00030;WAk0000000/<KZ00L0/<KZ0P0000060:^oj0030000002Z_^P0Z[kX00D0Z[kX0`0000060:Vm
j0800000102Y_NP200000080ZKgX00<000000:Vmj02Y_NP01@2Y_NP00`000000ZKgX0:Vmj0040000
0080ZKgX00<000000:Vmj02Y_NP01@2Y_NP2000000L0oooo0P0000040?ooo`800000103oool20000
00@0oooo000N0?ooo`8000006P3oool2000004@0oooo1`00000:0:K3k`030000002Va>l0Y/C_0080
00001@2Xanl2000000D0[Lc`0P0000040;KHl@80000000<0anc^0000003Fom40100000001@33jh`0
0000000000000000fOkE008000000`39in`2000000@0^M7/00<000000;36jP2`a^X00`2`a^X20000
00030:c1j@000000ZkoX00H0ZkoX00<000000:Znj02Z_^P0100000020:Vnj0030000002Y_NP0ZKgX
00D0ZKgX00<000000:Vmj02Y_NP00P0000040:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`0000060:Vm
j0030000002Y_NP0ZKgX00D0ZKgX0`0000050?ooo`040000003oool0oooo000000P0oooo00@00000
0?ooo`3oool000000`3oool001h0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0m
0?ooo`@0000000@0YL?_00000000000000000`2U`nl00`000000Y/?_0:K3k`070:K3k`<000002@2X
anl00`000000[Lc`0:g<l002000000P0]]Sa100000001P3Fom400000000000000000`n^<000000@0
fOkE0`0000070;WAk0030000002`a^X0/<KZ00800000102/`NT00`000000ZkoX0:^oj0040:^oj0<0
00001P2Y_^P00`000000ZKgX0:Vmj0050:Vmj0<000001`2Y_NP00`000000ZKgX0:Vmj00400000080
ZKgX00<000000:Vmj02Y_NP01@2Y_NP00`000000ZKgX0:Vmj00400000080ZKgX00<000000?ooo`3o
ool00`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000000<0oooo000N0?oo
o`030000003oool0oooo01`0oooo00<000000?ooo`3oool0>P3oool200000080YL?_0`0000070:G3
k`030000002V`nl0Y/?_00@0Y/?_0P0000020:K4k`030000002Xanl0Z<O_00L0Z<O_0`0000080;KH
l@8000000P37k>h00`000000e_oA00000004000000@0fOkE0P0000000`39in`000000;WAk0060;WA
k0<000001`2/`NT00`000000ZkoX0:^oj00300000080Z[kX00<000000:Vnj02Y_^P01@2Y_^P00`00
0000ZKgX0:Vmj004000000030:Vmj0000000000000L0ZKgX0`0000060:Vmj0030000002Y_NP0ZKgX
00D0ZKgX0`0000060:Vmj0800000103oool010000000oooo0?ooo`0000090?ooo`800000103oool0
01h0oooo0P00000N0?ooo`030000003oool0oooo03L0oooo0P0000020:G3k`@000002@2U`nl00`00
0000Y/?_0:K3k`03000000D0Y/C_00<000000:S7k`2Xanl0102Xanl200000080[Lc`00<000000;KH
l@2ff?401@2ff?401P000000anc^0<O/kP37k>h000000=Kod@@0000000<0`n^<0000003Io]D00P3I
o]D00`000000bNO/0<WWk0020<WWk0030000002idN`0^M7/00@000000P2`a^X00`000000[<7Y0:c1
j@050:c1j@<000001P2Z_^P00`000000ZKkX0:Vnj0040:Vnj0<000001P2Y_NP00`000000ZKgX0:Vm
j0030:Vmj08000000P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0030000002Y_NP0ZKgX00@000000P2Y
_NP00`000000ZKgX0:Vmj0050:Vmj08000000`3oool010000000oooo0?ooo`0000080?ooo`040000
003oool0oooo000000<0oooo000N0?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool0
<`3oool3000000040:G3k`0000000000000000@0YL?_00<000000:G3k`2U`nl01P2U`nl3000000T0
Y/C_00<000000:S7k`2Xanl00`0000050:g<l0030000002ff?40]]Sa0080]]Sa0P0000030<O/kP03
0000003Fom40e_oA00<0000000<0`n^<0000003Io]D00P3Io]D00`000000bNO/0<WWk0040<WWk080
00001P2`a^X00`000000[<7Y0:c1j@030:c1j@8000000P2[_nP00`000000Z[kX0:Znj0050:Znj003
0000002Y_^P0ZKkX00<000000P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0030000002Y_NP0ZKgX0080
0000102Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000001P2Y_NP00`000000ZKgX0:Vmj0050:Vmj080
0000103oool2000000X0oooo0P0000040?ooo`007@3oool00`000000oooo0?ooo`0P0?ooo`030000
003oool0oooo0300oooo0P000000102U`nl000000000000000080:G3k`040000002U`nl0YL?_0:G3
k`@000000P2V`nl00`000000Y/C_0:K4k`070:K4k`<000002@2]c?0010000000]]Sa000000000004
0<O/kP030000003Fom40e_oA00<0000000@0`n^<0000003Io]D0fOkE0P0000050<WWk080000000<0
^M7/0000002`a^X01P2`a^X00`000000[<7Y0:c1j@02000000@0ZkoX00<000000:Znj02Z_^P01@2Z
_^P3000000D0ZKgX00<000000:Vmj02Y_NP01@2Y_NP3000000L0ZKgX00<000000:Vmj02Y_NP00P2Y
_NP200000080ZKgX00<000000:Vmj02Y_NP01@2Y_NP00`000000ZKgX0:Vmj004000000030:Vmj000
0000000001D0oooo000M0?ooo`<000008@3oool00`000000oooo0?ooo`0/0?ooo`H000000P2U`nl0
0`000000YL?_0:G3k`060:G3k`@000001`2V`nl00`000000Y/C_0:K4k`040:K4k`8000000P2Xanl0
0`000000[Lc`0:g<l0080:g<l00300000037k>h0anc^00<0anc^00<000000=Kod@3Fom4020000005
0<WWk08000000`2idN`00`000000/<KZ0;36jP050;36jP<000001`2[_nP00`000000Z[kX0:Znj004
00000080ZKkX00<000000:Vmj02Y_NP0102Y_NP00`000000ZKgX0:Vmj004000000030:Vmj0000000
000000L0ZKgX00<000000:Vmj02Y_NP00P0000040:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`000006
0:Vmj0800000503oool001d0oooo00<000000?ooo`3oool08P3oool00`000000oooo0?ooo`0X0?oo
o`H000001@2U`nl00`000000YL?_0:G3k`040:G3k`8000000P2U`nl00`000000Y/?_0:K3k`070:K3
k`030000002Va>l0Y/C_00<000001@2Xanl00`000000[Lc`0:g<l0050:g<l080000000<0]]Sa0000
000000000P37k>h010000000e_oA0=Kod@3Fom4400000080`n^<00D000000=Wne@00000000000<WW
k002000000H0^M7/00<000000;36jP2`a^X0100000020:c1j@030000002[_nP0ZkoX00D0ZkoX0`00
00060:Vnj0030000002Y_NP0ZKgX00@0ZKgX0`0000060:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`00
00070:Vmj0030000002Y_NP0ZKgX00@0000000<0ZKgX0000002Y_NP01P2Y_NP2000001@0oooo000:
0?ooo`800000103oool2000000@0oooo0P0000050?ooo`030000003oool0oooo02<0oooo00<00000
0?ooo`3oool08`3oool7000000T0YL?_00<000000:G3k`2U`nl00`0000050:G3k`030000002V`nl0
Y/?_00H0Y/?_0`0000080:S7k`030000002]c?00[Lc`00<0[Lc`0P0000050;KHl@X0000000<0`n^<
0000003Io]D00`3Io]D00`000000^M7/0;WAk0070;WAk08000001`2/`NT00`000000ZkoX0:^oj004
00000080Z[kX00<000000:Vnj02Y_^P01@2Y_^P00`000000ZKgX0:Vmj00300000080ZKgX00<00000
0:Vmj02Y_NP0102Y_NP200000080ZKgX100000020:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`000006
0:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`00000C0?ooo`002@3oool010000000oooo0?ooo`000008
0?ooo`040000003oool0oooo000000@0oooo0P00000U0?ooo`030000003oool0oooo01h0oooo1000
0000102U`nl000000000000000030:G3k`030000002U`nl0YL?_00H0YL?_0`0000080:G3k`030000
002V`nl0Y/?_00@0Y/?_0P0000020:K4k`030000002Xanl0Z<O_00L0Z<O_00<000000:g<l02]c?00
0P0000060;KHl@8000000`37k>h2000000030<7YR@0000000000008000000`3Io]D2000000030<WW
k0000000^M7/00D0^M7/0P0000000`2`a^X000000:c1j@060:c1j@<000001P2Z_^P00`000000ZKkX
0:Vnj0050:Vnj0<000001@2Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000001P2Y_NP00`000000ZKgX
0:Vmj0030:Vmj080000000<0ZKgX0000002Y_NP01P2Y_NP00`000000ZKgX0:Vmj00400000080ZKgX
0P00000B0?ooo`002@3oool010000000oooo0?ooo`00000;0?ooo`030000003oool0oooo0080oooo
00<000000?ooo`3oool09@3oool00`000000oooo0?ooo`0K0?ooo`8000000P2U`nl3000000L0YL?_
00<000000:G3k`2U`nl00`2U`nl200000080YL?_00<000000:G3k`2U`nl01`2U`nl00`000000Y/?_
0:K3k`03000000D0Y/C_00<000000:S7k`2Xanl01P2Xanl3000000L0]]Sa00<000000<O/kP37k>h0
0P37k>h00`000000e_oA00000003000000@0fOkE00<000000<WWk039in`00P39in`200000080^M7/
0P0000030;36jP030000002/`NT0[<7Y00<0[<7Y0P0000020:^oj0030000002Z_^P0Z[kX00D0Z[kX
00<000000:Vnj02Y_^P0100000020:Vmj0030000002Y_NP0ZKgX00@0ZKgX00<000000:Vmj02Y_NP0
100000000`2Y_NP00000000000070:Vmj0030000002Y_NP0ZKgX00800000102Y_NP00`000000ZKgX
0:Vmj0040:Vmj0<000001P2Y_NP200000180oooo00090?ooo`040000003oool0oooo000000T0oooo
0P0000040?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool0603oool200000080YL?_
100000090:G3k`030000002U`nl0YL?_008000001@2U`nl00`000000YL?_0:G3k`060:G3k`<00000
2@2Va>l00`000000Z<O_0:S7k`030:S7k`8000000P2]c?02000000D0]]Sa00<000000<O/kP37k>h0
0P37k>h010000000e_oA00000031jHT2000000<0fOkE0P0000060<WWk0<000001P2`a^X00`000000
[<7Y0:c1j@02000000@0ZkoX00<000000:Znj02Z_^P01@2Z_^P3000000H0ZKgX00<000000:Vmj02Y
_NP0102Y_NP3000000H0ZKgX00<000000:Vmj02Y_NP01@2Y_NP3000000H0ZKgX00<000000:Vmj02Y
_NP0100000020:Vmj0030000002Y_NP0ZKgX00D0ZKgX0P00000A0?ooo`002@3oool010000000oooo
0?ooo`0000090?ooo`030000003oool0oooo00<0oooo0P00000Y0?ooo`030000003oool0oooo01@0
oooo0`000000102U`nl000000000000000040:G3k`030000002U`nl0YL?_00H0YL?_0`0000080:G3
k`030000002U`nl0YL?_00<0YL?_0P0000020:K3k`030000002Va>l0Y/C_00P0Y/C_00<000000:S7
k`2Xanl00P0000060:g<l080000000<0]]Sa0000000000001037k>h00`000000e_oA0=Kod@070000
00H0bNO/0P0000020;WAk0030000002`a^X0/<KZ00@0/<KZ0`0000070:^oj0030000002Z_^P0Z[kX
00@000000P2Y_^P00`000000ZKgX0:Vmj0050:Vmj0030000002Y_NP0ZKgX00<000000P2Y_NP00`00
0000ZKgX0:Vmj0040:Vmj08000000P2Y_NP400000080ZKgX00<000000:Vmj02Y_NP01@2Y_NP20000
00H0ZKgX00<000000:Vmj02Y_NP01@2Y_NP200000140oooo000:0?ooo`8000002P3oool3000000<0
oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0A0?ooo`80000000@0YL?_00000000
00000000202U`nl010000000YL?_0:G3k`2U`nl400000080YL?_00<000000:G3k`2U`nl01`2U`nl0
0`000000YL?_0:G3k`02000000D0Y/?_00<000000:K4k`2Va>l01`2Va>l3000000X0[Lc`00<00000
0<O/kP37k>h00`37k>h00`000000e_oA0=Kod@03000000<0`n^<00@000000=Wne@00000000000P39
in`2000000D0^M7/00<000000;36jP2`a^X00`0000020:c1j@030000002[_nP0ZkoX00H0ZkoX0P00
00060:Vnj0030000002Y_NP0ZKgX00D0ZKgX0`0000050:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`00
00060:Vmj0030000002Y_NP0ZKgX00<0ZKgX0P0000000`2Y_NP000000:Vmj0060:Vmj0030000002Y
_NP0ZKgX00@0000000<0ZKgX000000000000403oool001`0oooo00<000000?ooo`3oool0:P3oool0
0`000000oooo0?ooo`0=0?ooo`H000000P2U`nl00`000000YL?_0:G3k`060:G3k`@000001`2U`nl0
0`000000YL?_0:G3k`060:G3k`<00000202V`nl00`000000Y/C_0:K4k`040:K4k`8000000P2Xanl0
0`000000[Lc`0:g<l0060:g<l080000000<0]]Sa0000000000000P37k>h010000000e_oA0=Kod@3F
om4300000080`n^<00<000000=Wne@3Io]D00P3Io]D2000000L0^M7/0`0000060:c1j@030000002[
_nP0ZkoX00<0ZkoX0P0000000`2Z_^P000000:Vnj0060:Vnj0030000002Y_NP0ZKgX00@000000P2Y
_NP00`000000ZKgX0:Vmj0040:Vmj0030000002Y_NP0ZKgX00@0000000<0ZKgX0000000000001`2Y
_NP00`000000ZKgX0:Vmj002000000<0ZKgX00<000000:Vmj02Y_NP01@2Y_NP3000000H0ZKgX0P00
000?0?ooo`00703oool2000002`0oooo0P00000:0?ooo`H000001P2U`nl00`000000YL?_0:G3k`03
0:G3k`8000000P2U`nl00`000000YL?_0:G3k`070:G3k`040000002U`nl0YL?_0:G3k`@000000P2U
`nl00`000000Y/?_0:K3k`070:K3k`030000002Va>l0Y/C_00<000001@2Xanl00`000000[Lc`0:g<
l0030:g<l08000001@2ff?4800000080`n^<00@000000=Wne@3Io]D0fOkE0P0000000`39in`00000
0;WAk0040;WAk08000000P2`a^X00`000000[<7Y0:c1j@050:c1j@030000002[_nP0ZkoX00800000
0`2Z_^P00`000000ZKkX0:Vnj0050:Vnj0<000001P2Y_NP00`000000ZKgX0:Vmj0040:Vmj0<00000
1P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000001P2Y_NP00`000000ZKgX0:Vmj00400000080ZKgX
00<000000:Vmj02Y_NP0102Y_NP2000000l0oooo000L0?ooo`030000003oool0oooo02d0oooo00<0
00000?ooo`3oool00`3oool7000000X0YL?_00<000000:G3k`2U`nl00P0000050:G3k`030000002U
`nl0YL?_00H0YL?_100000070:G3k`030000002V`nl0Y/?_00H0Y/?_0`0000090:S7k`030000002]
c?00[Lc`008000001P2ff?42000000<0anc^100000000`33jh`000000=Wne@020=Wne@0300000039
in`0bNO/0080bNO/00<000000;WAk02idN`00P0000040;36jP030000002/`NT0[<7Y00D0[<7Y0`00
00060:Znj0030000002Y_^P0ZKkX00@000000P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0030000002Y
_NP0ZKgX00<000000P2Y_NP00`000000ZKgX0:Vmj0040:Vmj08000000P2Y_NP400000080ZKgX00<0
00000:Vmj02Y_NP0102Y_NP3000000H0ZKgX00<000000:Vmj02Y_NP0102Y_NP3000000h0oooo000K
0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`0000000P000000102U`nl0000000000000
00030:G3k`800000202U`nl3000000P0YL?_00<000000:G3k`2U`nl00`2U`nl200000080YL?_00<0
00000:G3k`2U`nl01`2U`nl010000000Y/?_0:K3k`2V`nl400000080Y/C_00<000000:S7k`2Xanl0
1`2Xanl3000000L0]]Sa00<000000<O/kP37k>h00P37k>h01P000000e_oA000000000000`n^<0000
0080fOkE0P0000060<WWk08000001`2`a^X00`000000[<7Y0:c1j@0400000080ZkoX00<000000:Zn
j02Z_^P0102Z_^P3000000H0ZKgX00<000000:Vmj02Y_NP01@2Y_NP3000000D0ZKgX00<000000:Vm
j02Y_NP01@2Y_NP3000000H0ZKgX00<000000:Vmj02Y_NP0100000020:Vmj0030000002Y_NP0ZKgX
00D0ZKgX00<000000:Vmj02Y_NP00`0000020:Vmj08000003@3oool001/0oooo0P00000`0?ooo`80
00000P2U`nl2000000P0YL?_00@000000:G3k`2U`nl0YL?_100000020:G3k`030000002U`nl0YL?_
00L0YL?_00<000000:G3k`2U`nl00P0000050:G3k`030000002U`nl0YL?_00H0YL?_100000070:K4
k`030000002Xanl0Z<O_00@0Z<O_0P0000020:g<l0030000002ff?40]]Sa00<0]]Sa0P0000040<O/
kP030000003Fom40000000<0000000<0fOkE0000000000001P39in`2000000030;WAk0000000/<KZ
00H0/<KZ0`0000070:^oj0030000002Z_^P0Z[kX00<000000P2Y_^P00`000000ZKgX0:Vmj0050:Vm
j0030000002Y_NP0ZKgX00@0000000<0ZKgX0000002Y_NP01P2Y_NP00`000000ZKgX0:Vmj0040000
00030:Vmj0000000000000H0ZKgX0`0000060:Vmj0030000002Y_NP0ZKgX00D0ZKgX0`0000050:Vm
j08000001`3oool500000040oooo000K0?ooo`030000003oool0oooo02d0oooo0P0000001P2U`nl0
0000000000000000YL?_000000P0YL?_100000070:G3k`030000002U`nl0YL?_00H0YL?_0`000008
0:G3k`030000002U`nl0YL?_00<0YL?_0P0000020:K3k`030000002Va>l0Y/C_00L0Y/C_00<00000
0:S7k`2Xanl00`0000050:g<l080000000<0]]Sa0000000000001037k>h2000000030=Kod@000000
000000D0000000<0fOkE0000000000000P39in`2000000@0^M7/0P0000020;36jP<000000P2/`NT0
0`000000ZkoX0:^oj0050:^oj0<000001@2Y_^P00`000000ZKgX0:Vmj0050:Vmj0<000001P2Y_NP0
0`000000ZKgX0:Vmj0040:Vmj0<000001@2Y_NP00`000000ZKgX000000020:Vmj0<000000P2Y_NP0
0`000000ZKgX0:Vmj0050:Vmj0030000002Y_NP0ZKgX00@0000000<0ZKgX0000002Y_NP01P2Y_NP2
000000P0oooo00<000000?ooo`3oool00@3oool001/0oooo00<000000?ooo`3oool0:P3oool60000
00D0YL?_00<000000:G3k`2U`nl00`2U`nl200000080YL?_00<000000:G3k`2U`nl01`2U`nl01000
0000YL?_0:G3k`2U`nl400000080YL?_00<000000:G3k`2U`nl01`2U`nl00`000000YL?_0:G3k`02
000000D0Y/?_00<000000:K4k`2Va>l01P2Va>l3000000X0[Lc`00<000000<O/kP37k>h00`37k>h0
0`000000e_oA0=Kod@0400000080`n^<00<000000=Wne@3Io]D00P3Io]D2000000L0^M7/0`000006
0:c1j@030000002[_nP0ZkoX00@000000P2Z_^P00`000000ZKkX0:Vnj0040:Vnj0030000002Y_NP0
ZKgX00@000000P2Y_NP00`000000ZKgX0:Vmj0040:Vmj0030000002Y_NP0ZKgX00@000000P2Y_NP0
0`000000ZKgX0:Vmj0040:Vmj0@000001@2Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000001P2Y_NP0
0`000000ZKgX0:Vmj0040:Vmj0<000001`3oool00`000000oooo0?ooo`010?ooo`006`3oool20000
02L0oooo1`0000090:G3k`030000002U`nl0YL?_008000001@2U`nl00`000000YL?_0:G3k`060:G3
k`@000001`2U`nl00`000000YL?_0:G3k`060:G3k`<00000202V`nl00`000000Y/C_0:K4k`030:K4
k`8000000P2Xanl00`000000[Lc`0:g<l0060:g<l080000000<0]]Sa0000000000000P37k>h01000
0000e_oA0=Kod@3Fom44000000030<?[S0000000fOkE0080fOkE0P0000000`39in`000000;WAk004
0;WAk08000000P2`a^X00`000000[<7Y0:c1j@050:c1j@8000001P2Z_^P00`000000ZKkX0:Vnj004
0:Vnj0<000001P2Y_NP00`000000ZKgX0:Vmj0040:Vmj0<000001P2Y_NP00`000000ZKgX0:Vmj003
0:Vmj0@000001`2Y_NP00`000000ZKgX0:Vmj004000000030:Vmj0000000ZKgX00H0ZKgX00<00000
0:Vmj02Y_NP0100000000`2Y_NP00000000000070?ooo`030000003oool0oooo0040oooo000K0?oo
o`030000003oool0oooo02<0oooo0`000000102U`nl000000000000000030:G3k`8000001`2U`nl3
000000P0YL?_00<000000:G3k`2U`nl00`2U`nl200000080YL?_00<000000:G3k`2U`nl01`2U`nl0
10000000YL?_0:G3k`2U`nl400000080YL?_00<000000:K3k`2V`nl01`2V`nl00`000000Y/C_0:K4
k`02000000D0Z<O_0P0000040:g<l08000001@2ff?49000000030<?[S0000000fOkE0080fOkE00<0
00000<WWk039in`00P39in`00`000000^M7/0;WAk002000000@0/<KZ00<000000:c1j@2/`NT00`2/
`NT2000000030:^oj0000000Z[kX00H0Z[kX00<000000:Vnj02Y_^P00`0000020:Vmj0030000002Y
_NP0ZKgX00D0ZKgX00<000000:Vmj02Y_NP00`0000020:Vmj0030000002Y_NP0ZKgX00D0ZKgX00<0
00000:Vmj02Y_NP00P0000030:Vmj08000001P2Y_NP3000000H0ZKgX00<000000:Vmj02Y_NP0102Y
_NP3000000H0ZKgX0P0000060?ooo`030000003oool0oooo0040oooo00020?ooo`800000103oool2
000000D0oooo00<000000?ooo`3oool00P3oool2000000@0oooo00<000000?ooo`3oool08P3oool2
000000040:G3k`0000000000000000P0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002U
`nl0YL?_00L0YL?_00<000000:G3k`2U`nl00P0000050:G3k`030000002U`nl0YL?_00H0YL?_1000
00070:G3k`030000002V`nl0Y/?_00H0Y/?_0`0000090:S7k`040000002]c?000000000000H0]]Sa
0P0000030<O/kPH000000P3Io]D2000000H0bNO/0P0000070;36jP030000002/`NT0[<7Y00800000
0`2[_nP00`000000Z[kX0:Znj0050:Znj0<000001@2Y_NP00`000000ZKgX0:Vmj0050:Vmj0<00000
1@2Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000001@2Y_NP200000080ZKgX100000020:Vmj0030000
002Y_NP0ZKgX00@0ZKgX00<000000:Vmj02Y_NP0100000020:Vmj0030000002Y_NP0ZKgX00@0ZKgX
0P0000050?ooo`8000000`3oool000050?ooo`000000oooo0?ooo`0000002P3oool01@000000oooo
0?ooo`3oool000000080oooo00@000000?ooo`3oool0oooo0P00000Q0?ooo`D000000P2U`nl00`00
0000YL?_0:G3k`060:G3k`<000001`2U`nl2000000L0YL?_0`0000080:G3k`030000002U`nl0YL?_
00<0YL?_0P0000020:G3k`030000002U`nl0YL?_00L0YL?_00@000000:K3k`2V`nl0Y/?_10000002
0:K4k`030000002Xanl0Z<O_00L0Z<O_0P0000070;KHl@0300000037k>h0anc^0080anc^00<00000
0=Kod@0000000`0000000`3Io]D00000000000060<WWk080000000<0^M7/0000002`a^X01P2`a^X3
000000H0ZkoX00<000000:Znj02Z_^P0100000020:Vnj0030000002Y_NP0ZKgX00@0ZKgX00<00000
0:Vmj02Y_NP0100000020:Vmj0030000002Y_NP0ZKgX00@0ZKgX00<000000:Vmj02Y_NP010000002
0:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`0000060:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`000006
0:Vmj0030000002Y_NP0ZKgX00@0ZKgX0P00000:0?ooo`0000D0oooo0000003oool0oooo0000000:
0?ooo`030000003oool0oooo00@0oooo00D000000?ooo`3oool0oooo0000000O0?ooo`D000001P2U
`nl00`000000YL?_0:G3k`030:G3k`8000000P2U`nl00`000000YL?_0:G3k`070:G3k`030000002U
`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3k`070:G3k`030000002U`nl0YL?_008000001@2U
`nl00`000000YL?_0:G3k`060:G3k`@000001`2Va>l00`000000Z<O_0:S7k`040:S7k`8000000P2]
c?000`000000]]Sa0;KHl@030;KHl@0300000037k>h0anc^0080anc^00<000000=Kod@0000001P00
00000`3Io]D00000000000020<WWk0800000102idN`00`000000/<KZ0;36jP0400000080[<7Y00<0
00000:^oj02[_nP0102[_nP3000000H0ZKkX00<000000:Vmj02Y_NP0102Y_NP3000000H0ZKgX00<0
00000:Vmj02Y_NP0102Y_NP3000000L0ZKgX00<000000:Vmj02Y_NP00`0000000`2Y_NP000000000
00070:Vmj0030000002Y_NP0ZKgX00<000000P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0030000002Y
_NP0ZKgX00<0000000<0oooo0000003oool02@3oool000050?ooo`000000oooo0?ooo`0000002`3o
ool010000000oooo0?ooo`3oool2000000@0oooo00<000000?ooo`3oool06@3oool7000000T0YL?_
00<000000:G3k`2U`nl00P0000050:G3k`030000002U`nl0YL?_00H0YL?_0`0000070:G3k`800000
1`2U`nl3000000P0YL?_00<000000:G3k`2U`nl00`2U`nl200000080Y/?_00<000000:K4k`2Va>l0
1`2Va>l00`000000Z<O_0:S7k`03000000D0[Lc`0P0000000`2ff?400000000000040<O/kP030000
003Fom40e_oA00<000000P33jh`00`000000fOkE0=Wne@020=Wne@8000001P2idN`3000000L0[<7Y
00<000000:^oj02[_nP00`0000020:Znj0030000002Y_^P0ZKkX00D0ZKkX00<000000:Vmj02Y_NP0
0`0000020:Vmj0030000002Y_NP0ZKgX00D0ZKgX00<000000:Vmj02Y_NP00`0000020:Vmj0030000
002Y_NP0ZKgX00D0ZKgX0`0000040:Vmj08000001`2Y_NP3000000D0ZKgX00<000000:Vmj02Y_NP0
1@2Y_NP3000000D0oooo00<000000?ooo`3oool01`3oool000050?ooo`000000oooo0?ooo`000000
203oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00<0oooo0P00000G0?ooo`<0
000000@0YL?_00000000000000000`2U`nl2000000L0YL?_0`0000080:G3k`030000002U`nl0YL?_
00<0YL?_0P0000020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U`nl0100000020:G3
k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U`nl00P0000050:K3k`030000002Va>l0Y/C_
00H0Y/C_0`00000:0:g<l00300000037k>h0anc^00<0anc^00<000000=Kod@3Fom401P0000030=Wn
e@80000000<0bNO/0000000000000P2idN`300000080/<KZ00<000000:c1j@2/`NT01@2/`NT30000
00D0Z[kX00<000000:Vnj02Y_^P01@2Y_^P3000000D0ZKgX00<000000:Vmj02Y_NP01@2Y_NP30000
00H0ZKgX00<000000:Vmj02Y_NP0100000000`2Y_NP000000:Vmj0050:Vmj08000000P2Y_NP40000
00030:Vmj0000000ZKgX00H0ZKgX00<000000:Vmj02Y_NP0100000080?ooo`030000003oool0oooo
00L0oooo00020?ooo`8000002@3oool4000000<0oooo0`0000030?ooo`030000003oool0oooo0180
oooo10000000102U`nl000000000000000080:G3k`030000002U`nl0YL?_00@000000P2U`nl00`00
0000YL?_0:G3k`070:G3k`030000002U`nl0YL?_008000001@2U`nl00`000000YL?_0:G3k`060:G3
k`<000001`2U`nl2000000L0YL?_0`0000080:K3k`030000002Va>l0Y/C_00<0Y/C_0P0000020:S7
k`030000002]c?00[Lc`00H0[Lc`0P0000000`2ff?400000000000020<O/kP040000003Fom40e_oA
0=Kod@@00000103Io]D00`000000bNO/0<WWk0030<WWk08000001P2`a^X200000080[<7Y0`000002
0:^oj0030000002Z_^P0Z[kX00@0Z[kX00<000000:Vnj02Y_^P0100000020:Vmj0030000002Y_NP0
ZKgX00@0ZKgX00<000000:Vmj02Y_NP0100000020:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`000006
0:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`0000060:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`00000<
0?ooo`030000003oool0oooo00L0oooo000I0?ooo`030000003oool0oooo0100oooo0`0000020:G3
k`<000000P2U`nl2000000L0YL?_0`0000070:G3k`8000001`2U`nl3000000P0YL?_00<000000:G3
k`2U`nl00`2U`nl200000080YL?_00<000000:G3k`2U`nl01`2U`nl00`000000YL?_0:G3k`040000
0080YL?_00<000000:K3k`2V`nl01`2V`nl00`000000Y/C_0:K4k`02000000D0Z<O_00<000000:g<
l02]c?000`2]c?02000000D0]]Sa2@0000030=Wne@8000001039in`2000000030;WAk0000000/<KZ
00H0/<KZ0`0000050:^oj0030000002Z_^P0Z[kX00@0Z[kX0`0000060:Vmj0030000002Y_NP0ZKgX
00@0ZKgX0`0000070:Vmj0030000002Y_NP0ZKgX00<000000P2Y_NP00`000000ZKgX0:Vmj0040:Vm
j0030000002Y_NP0ZKgX00@0000000<0ZKgX0000000000001P2Y_NP00`000000ZKgX0:Vmj0040000
00l0oooo00<000000?ooo`3oool01`3oool001T0oooo0P00000?0?ooo`8000000P2U`nl3000000L0
YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U
`nl0100000020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U`nl00P0000050:G3k`03
0000002U`nl0YL?_00H0YL?_0`0000070:G3k`030000002V`nl0Y/?_00H0Y/?_0`0000080:S7k`03
0000002]c?00[Lc`008000001P2ff?42000000<0anc^2@0000030<WWk08000000`2idN`00`000000
/<KZ0;36jP030;36jP8000000P2/`NT00`000000ZkoX0:^oj0040:^oj0030000002Z_^P0Z[kX00<0
00000P2Y_^P00`000000ZKgX0:Vmj0050:Vmj0030000002Y_NP0ZKgX00<000000P2Y_NP00`000000
ZKgX0:Vmj0050:Vmj0<000001@2Y_NP00`000000ZKgX0:Vmj0040:Vmj0<000001@2Y_NP2000000H0
ZKgX0`00000C0?ooo`030000003oool0oooo00L0oooo000I0?ooo`030000003oool0oooo00`0oooo
0P0000001P2U`nl00000000000000000YL?_000000P0YL?_0`0000070:G3k`8000001`2U`nl30000
00L0YL?_0P0000070:G3k`<000001`2U`nl00`000000YL?_0:G3k`040:G3k`8000000P2U`nl00`00
0000YL?_0:G3k`060:G3k`040000002V`nl0Y/?_0:K3k`@000000P2Va>l00`000000Z<O_0:S7k`06
0:S7k`<000001`2ff?400`000000anc^0<O/kP020<O/kP040000003Fom400000000000<0`n^<00<0
00000=Wne@3Io]D00`0000060;WAk0030000002`a^X0/<KZ00800000102/`NT00`000000ZkoX0:^o
j0040:^oj0<000001P2Y_^P00`000000ZKgX0:Vmj0040:Vmj0<000001P2Y_NP00`000000ZKgX0:Vm
j004000000030:Vmj0000000ZKgX00H0ZKgX00<000000:Vmj02Y_NP00`0000020:Vmj0030000002Y
_NP0ZKgX00@0ZKgX0P0000020:Vmj0<000005P3oool00`000000oooo0?ooo`070?ooo`006@3oool0
0`000000oooo0?ooo`090?ooo`H000001@2U`nl00`000000YL?_0:G3k`030:G3k`8000000P2U`nl0
0`000000YL?_0:G3k`070:G3k`030000002U`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3k`07
0:G3k`030000002U`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3k`060:G3k`030000002U`nl0
YL?_00<000001@2U`nl00`000000YL?_0:G3k`050:G3k`@000001`2Va>l00`000000Z<O_0:S7k`03
0:S7k`8000000P2]c?02000000D0]]Sa00<000000<O/kP37k>h00P37k>h00`000000e_oA00000002
00000080`n^<00<000000=Wne@3Io]D00P0000000`39in`000000;WAk0050;WAk0<000001`2/`NT0
0`000000ZkoX0:^oj00300000080Z[kX00<000000:Vnj02Y_^P01@2Y_^P00`000000ZKgX0:Vmj003
00000080ZKgX00<000000:Vmj02Y_NP0102Y_NP3000000H0ZKgX00<000000:Vmj02Y_NP0102Y_NP3
000000D0ZKgX00<000000:Vmj02Y_NP01@2Y_NP3000001T0oooo00<000000?ooo`3oool01`3oool0
01T0oooo0P0000060?ooo`L000002@2U`nl00`000000YL?_0:G3k`02000000D0YL?_00<000000:G3
k`2U`nl01P2U`nl3000000L0YL?_0P0000070:G3k`<000001`2U`nl00`000000YL?_0:G3k`050:G3
k`<00000202U`nl00`000000YL?_0:G3k`030:G3k`8000000P2V`nl00`000000Y/C_0:K4k`070:K4
k`030000002Xanl0Z<O_008000001P2]c?02000000030;KHl@000000000000@0anc^00<000000=Ko
d@0000000P0000000`33jh`00000000000020=Wne@0300000039in`0bNO/0080bNO/00<000000;WA
k02idN`00`0000020;36jP030000002/`NT0[<7Y00D0[<7Y0`0000050:Znj0030000002Y_^P0ZKkX
00D0ZKkX0`0000060:Vmj0030000002Y_NP0ZKgX00<000000P2Y_NP00`000000ZKgX0:Vmj0040:Vm
j0030000002Y_NP0ZKgX00@000000P2Y_NP00`000000ZKgX0:Vmj0040:Vmj0030000002Y_NP0ZKgX
00D000006`3oool00`000000oooo0?ooo`070?ooo`006@3oool00`000000oooo0?ooo`020?ooo`<0
00000P2U`nl2000000<0YL?_0P0000070:G3k`<00000202U`nl00`000000YL?_0:G3k`030:G3k`80
00000P2U`nl00`000000YL?_0:G3k`070:G3k`030000002U`nl0YL?_00@000000P2U`nl00`000000
YL?_0:G3k`060:G3k`030000002U`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3k`070:G3k`03
0000002U`nl0YL?_008000001@2V`nl00`000000Y/C_0:K4k`060:K4k`<000002P2]c?000`000000
anc^0<O/kP030<O/kP030000003Fom40e_oA00@000000P3Io]D2000000H0bNO/0P0000060;36jP80
00000P2/`NT300000080ZkoX00<000000:Znj02Z_^P0102Z_^P00`000000ZKkX0:Vnj00400000080
ZKgX00<000000:Vmj02Y_NP0102Y_NP3000000D0ZKgX00<000000:Vmj02Y_NP0102Y_NP3000000H0
ZKgX00<000000:Vmj02Y_NP0102Y_NP3000000D0oooo00<000000?ooo`3oool06@3oool00`000000
oooo0?ooo`060?ooo`00603oool010000000oooo0?ooo`3oool200000080YL?_0`0000070:G3k`03
0000002U`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3k`070:G3k`030000002U`nl0YL?_0080
00001@2U`nl00`000000YL?_0:G3k`060:G3k`<000001`2U`nl00`000000YL?_0:G3k`050:G3k`<0
00001`2U`nl2000000L0YL?_0`0000080:K3k`030000002Va>l0Y/C_00<0Y/C_0P0000020:S7k`03
0000002]c?00[Lc`00H0[Lc`0P0000000`2ff?400000000000020<O/kP030000003Fom40000000@0
000000<0fOkE0000000000001P39in`2000000030;WAk0000000/<KZ00H0/<KZ0`0000060:^oj003
0000002Z_^P0Z[kX0080Z[kX100000060:Vmj0030000002Y_NP0ZKgX00@000000P2Y_NP00`000000
ZKgX0:Vmj0040:Vmj0030000002Y_NP0ZKgX00<000000P2Y_NP00`000000ZKgX0:Vmj0050:Vmj003
0000002Y_NP0ZKgX00<000002@3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00H0
oooo000H0?ooo`050000003oool000000000002U`nl00`0000000`2U`nl000000:G3k`070:G3k`<0
00001`2U`nl2000000L0YL?_0`0000080:G3k`030000002U`nl0YL?_00<0YL?_0P0000020:G3k`03
0000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00L0
YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002V`nl0Y/?_00L0Y/?_00<000000:K4k`2V
a>l00P0000050:S7k`030000002]c?00[Lc`00<0[Lc`0P0000050;KHl@/0000000<0fOkE00000000
00000P39in`2000000@0^M7/00<000000;36jP2`a^X0100000020:c1j@030000002[_nP0ZkoX00@0
ZkoX1@0000030:Vnj0030000002Y_NP0ZKgX00@0ZKgX0`0000060:Vmj0030000002Y_NP0ZKgX00@0
ZKgX0`0000060:Vmj0030000002Y_NP0ZKgX0080ZKgX1@00000=0?ooo`030000003oool0oooo01L0
oooo00<000000?ooo`3oool01P3oool001P0oooo1@0000050:G3k`030000002U`nl0YL?_00<0YL?_
0P0000020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000
002U`nl0YL?_00L0YL?_00<000000:G3k`2U`nl00P0000050:G3k`030000002U`nl0YL?_00D0YL?_
0`0000070:G3k`8000001`2U`nl3000000L0YL?_0P0000070:K3k`<00000202Xanl00`000000[Lc`
0:g<l002000000H0]]Sa0P0000020<O/kP@000000P33jh`00`000000fOkE0=Wne@020=Wne@800000
1P2idN`3000000L0[<7Y00<000000:^oj02[_nP00`0000020:Znj0030000002Y_^P0ZKkX00@0ZKkX
00<000000:Vmj02Y_NP00`0000020:Vmj0030000002Y_NP0ZKgX00D0ZKgX00<000000:Vmj02Y_NP0
0`0000020:Vmj0030000002Y_NP0ZKgX00@0ZKgX1@00000B0?ooo`030000003oool0oooo01L0oooo
00<000000?ooo`3oool01P3oool00100oooo1@0000030?ooo`<00000202U`nl00`000000YL?_0:G3
k`02000000D0YL?_0P0000070:G3k`<000001`2U`nl2000000L0YL?_0`0000070:G3k`030000002U
`nl0YL?_00<0YL?_0P0000020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U`nl01000
00020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:K3k`2V`nl0100000020:K4k`030000002X
anl0Z<O_00H0Z<O_0`0000070;KHl@0400000037k>h0anc^0<O/kPD0000000<0`n^<0000003Io]D0
0P3Io]D2000000030<WWk000000000000080^M7/0`0000020;36jP030000002/`NT0[<7Y00D0[<7Y
0`0000050:Znj0030000002Y_^P0ZKkX00@0ZKkX0`0000050:Vmj0030000002Y_NP0ZKgX00D0ZKgX
0`0000060:Vmj0030000002Y_NP0ZKgX00<00000603oool00`000000oooo0?ooo`0F0?ooo`030000
003oool0oooo00H0oooo000B0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`0000001`2U
`nl3000000T0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00L0YL?_00<0
00000:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl00P00
00050:G3k`030000002U`nl0YL?_00H0YL?_0`0000070:G3k`030000002U`nl0YL?_00H0YL?_0`00
00070:K4k`030000002Xanl0Z<O_00<0Z<O_0P0000020:g<l0030000002ff?40]]Sa00<0]]Sa0P00
00020<O/kP80000000H0e_oA00000000000000000<?[S00000030=Wne@0300000039in`0bNO/00<0
bNO/0P0000060;36jP8000000P2/`NT300000080ZkoX00<000000:Znj02Z_^P0102Z_^P00`000000
ZKkX0:Vnj00300000080ZKgX00<000000:Vmj02Y_NP0102Y_NP00`000000ZKgX0:Vmj00400000080
ZKgX00<000000:Vmj02Y_NP0102Y_NP3000001`0oooo00<000000?ooo`3oool05@3oool00`000000
oooo0?ooo`060?ooo`004P3oool00`000000oooo0?ooo`050?ooo`@000000P2U`nl300000080YL?_
00<000000:G3k`2U`nl01`2U`nl3000000L0YL?_0P0000070:G3k`<000001`2U`nl00`000000YL?_
0:G3k`050:G3k`<00000202U`nl00`000000YL?_0:G3k`030:G3k`8000000P2U`nl00`000000YL?_
0:G3k`060:G3k`040000002U`nl0YL?_0:G3k`@000000P2V`nl00`000000Y/C_0:K4k`060:K4k`03
0000002Xanl0Z<O_008000001@2]c?02000000030;KHl@000000000000<0anc^00<000000=Kod@3F
om401@0000020=Wne@8000001039in`2000000030;WAk0000000/<KZ00H0/<KZ0`0000050:^oj003
0000002Z_^P0Z[kX00@0Z[kX0`0000050:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`0000070:Vmj003
0000002Y_NP0ZKgX00<000007`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00H0
oooo000B0?ooo`030000003oool0oooo00L0oooo00@000000?ooo`00000000001P2U`nl2000000D0
YL?_0P0000020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U`nl0100000020:G3k`03
0000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00L0
YL?_00<000000:G3k`2U`nl00P0000050:G3k`030000002U`nl0YL?_00D0YL?_100000070:K3k`03
0000002Va>l0Y/C_00D0Y/C_0`0000080:g<l0<000000`37k>h00`000000e_oA0000000500000004
0=Wne@0000000000000000<0bNO/0P0000040;WAk0030000002`a^X0/<KZ00@000000P2/`NT00`00
0000ZkoX0:^oj0040:^oj0030000002Z_^P0Z[kX00<000000P2Y_^P00`000000ZKgX0:Vmj0040:Vm
j0030000002Y_NP0ZKgX00<000000P2Y_NP00`000000ZKgX0:Vmj0050:Vmj0<000008`3oool00`00
0000oooo0?ooo`0E0?ooo`030000003oool0oooo00D0oooo000B0?ooo`030000003oool0oooo00P0
oooo00<000000?ooo`000000202U`nl00`000000YL?_0:G3k`02000000D0YL?_00<000000:G3k`2U
`nl01P2U`nl3000000L0YL?_00<000000:G3k`2U`nl01@2U`nl3000000L0YL?_0P0000070:G3k`<0
00001`2U`nl00`000000YL?_0:G3k`030:G3k`8000000P2U`nl00`000000Y/?_0:K3k`070:K3k`03
0000002Va>l0Y/C_00@000000P2Xanl00`000000[Lc`0:g<l0040:g<l08000000`2ff?4600000003
0=Kod@000000000000@000000P3Io]D3000000L0^M7/0P0000060:c1j@030000002[_nP0ZkoX00@0
ZkoX0`0000060:Vnj0030000002Y_NP0ZKgX0080ZKgX100000060:Vmj0030000002Y_NP0ZKgX00@0
00009`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00D0oooo000A0?ooo`800000
2`3oool010000000oooo0000000000060:G3k`<00000202U`nl00`000000YL?_0:G3k`030:G3k`80
00000P2U`nl00`000000YL?_0:G3k`060:G3k`030000002U`nl0YL?_00@000000P2U`nl00`000000
YL?_0:G3k`070:G3k`030000002U`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3k`060:G3k`03
0000002U`nl0YL?_008000001@2U`nl2000000L0Y/?_0`0000070:S7k`8000000P2]c?02000000D0
]]Sa0P0000020<O/kP030000003Fom400000008000000P33jh`00`000000fOkE0=Wne@0200000003
0<WWk0000000^M7/00@0^M7/0P0000000`2`a^X000000:c1j@060:c1j@030000002[_nP0ZkoX00<0
00000P2Z_^P00`000000ZKkX0:Vnj0040:Vnj0D000000`2Y_NP00`000000ZKgX0:Vmj0040:Vmj0<0
0000;03oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00D0oooo000O0?ooo`800000
00@0oooo0000002U`nl0YL?_0`0000020:G3k`030000002U`nl0YL?_00L0YL?_00<000000:G3k`2U
`nl00P0000050:G3k`030000002U`nl0YL?_00H0YL?_0P0000070:G3k`8000001`2U`nl3000000L0
YL?_00<000000:G3k`2U`nl01@2U`nl3000000T0YL?_00<000000:K3k`2V`nl0100000020:K4k`03
0000002Xanl0Z<O_00H0Z<O_0`0000060;KHl@0500000037k>h0anc^0<O/kP0000000P3Fom430000
00050<?[S0000000fOkE0=Wne@0000001039in`00`000000^M7/0;WAk002000000@0/<KZ00<00000
0:c1j@2/`NT0102/`NT3000000H0Z[kX00<000000:Vnj02Y_^P00`0000020:Vmj0030000002Y_NP0
ZKgX00@0ZKgX00<000000:Vmj02Y_NP00`00000_0?ooo`030000003oool0oooo01<0oooo00<00000
0?ooo`3oool01@3oool00240oooo00@000000?ooo`00000000001P2U`nl2000000L0YL?_0`000008
0:G3k`030000002U`nl0YL?_00<0YL?_0P0000000`2U`nl000000:G3k`080:G3k`030000002U`nl0
YL?_00@000000P2U`nl00`000000YL?_0:G3k`060:G3k`030000002U`nl0YL?_00@000000P2U`nl0
0`000000YL?_0:G3k`070:G3k`<000001`2Va>l00`000000Z<O_0:S7k`030:S7k`8000000P2]c?00
0`000000]]Sa0;KHl@030;KHl@0500000037k>h0anc^0<O/kP0000000P3Fom43000000030<?[S000
0000fOkE008000001P39in`2000000L0/<KZ00<000000:c1j@2/`NT00`0000020:^oj0030000002Z
_^P0Z[kX00@0Z[kX0`0000050:Vmj0030000002Y_NP0ZKgX00@0ZKgX0`00000c0?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool01@3oool00280oooo00<000000?ooo`000000202U`nl0
0`000000YL?_0:G3k`0400000080YL?_00<000000:G3k`2U`nl01`2U`nl00`000000YL?_0:G3k`02
000000@0YL?_0P0000070:G3k`<000001`2U`nl00`000000YL?_0:G3k`050:G3k`<000001`2U`nl2
000000D0YL?_0P0000020:K3k`030000002Va>l0Y/C_00H0Y/C_00<000000:S7k`2Xanl00P000005
0:g<l080000000<0]]Sa0000000000001037k>h00`000000e_oA00000002000000040<?[S0000000
0000000000H0bNO/0P0000000`2idN`000000;36jP060;36jP<000001P2[_nP00`000000Z[kX0:Zn
j00300000080ZKkX00<000000:Vmj02Y_NP0102Y_NP00`000000ZKgX0:Vmj003000003L0oooo00<0
00000?ooo`3oool04@3oool00`000000oooo0?ooo`050?ooo`008`3oool010000000oooo00000000
00060:G3k`<000001`2U`nl2000000L0YL?_0`0000080:G3k`030000002U`nl0YL?_00@000000P2U
`nl00`000000YL?_0:G3k`060:G3k`030000002U`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3
k`070:G3k`030000002U`nl0YL?_008000001@2V`nl00`000000Y/C_0:K4k`050:K4k`<000002@2]
c?000`000000anc^0<O/kP030<O/kP030000003Fom40e_oA00H0000000<0fOkE0000000000000P39
in`2000000@0^M7/0P0000020;36jP<000000P2/`NT00`000000ZkoX0:^oj0040:^oj0<000001@2Y
_^P00`000000ZKgX0:Vmj0040:Vmj0<00000>`3oool00`000000oooo0?ooo`0@0?ooo`030000003o
ool0oooo00D0oooo000T0?ooo`040000003oool0oooo00000080YL?_0`0000020:G3k`030000002U
`nl0YL?_00L0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00H0YL?_0`00
00070:G3k`8000001P2U`nl3000000L0YL?_0P0000070:G3k`<00000202V`nl00`000000Y/C_0:K4
k`0400000080Z<O_00<000000:g<l02]c?001@2]c?02000000030;KHl@00000000000080anc^00<0
00000=Kod@000000100000000`33jh`000000=Wne@030=Wne@8000001`2idN`3000000H0[<7Y00<0
00000:^oj02[_nP00`0000020:Znj0030000002Y_^P0ZKkX00@0ZKkX00<000000:Vmj02Y_NP00`00
000n0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0103oool002D0oooo0P000000
0`3oool00000000000060:G3k`8000001`2U`nl3000000L0YL?_00<000000:G3k`2U`nl00`2U`nl2
00000080YL?_00<000000:G3k`2U`nl01`2U`nl00`000000YL?_0:G3k`0300000080YL?_00<00000
0:G3k`2U`nl01`2U`nl00`000000YL?_0:G3k`0400000080YL?_00<000000:K3k`2V`nl01P2V`nl3
000000L0Z<O_00<000000:g<l02]c?000P2]c?02000000D0]]Sa2P0000030=Wne@80000000<0bNO/
0000000000000`2idN`300000080/<KZ00<000000:c1j@2/`NT0102/`NT3000000D0Z[kX00<00000
0:Vnj02Y_^P0102Y_^P300000480oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`04
0?ooo`009`3oool00`000000oooo000000080:G3k`030000002U`nl0YL?_00@000000P2U`nl00`00
0000YL?_0:G3k`060:G3k`030000002U`nl0YL?_008000001@2U`nl2000000L0YL?_0`0000060:G3
k`8000001`2U`nl3000000L0YL?_00<000000:K3k`2V`nl00`2V`nl200000080Y/C_00<000000:S7
k`2Xanl01P2Xanl010000000[Lc`0000000000060;KHl@8000000P37k>h00`000000e_oA00000002
000000050<?[S0000000fOkE0=Wne@0000001@39in`3000000H0/<KZ00<000000:c1j@2/`NT00`00
00020:^oj0030000002Z_^P0Z[kX00@0Z[kX00<000000:Vnj02Y_^P00`0000160?ooo`030000003o
ool0oooo00l0oooo00<000000?ooo`3oool0103oool002P0oooo00@000000?ooo`00000000001P2U
`nl3000000L0YL?_0P0000060:G3k`<000002@2U`nl00`000000YL?_0:G3k`0400000080YL?_00<0
00000:G3k`2U`nl01P2U`nl00`000000YL?_0:G3k`0400000080YL?_00<000000:G3k`2U`nl01P2U
`nl00`000000Y/?_0:K3k`02000000D0Y/C_00<000000:S7k`2Xanl01@2Xanl2000000H0]]Sa0P00
00030<O/kP030000003Fom40000000@0000000<0fOkE0000000000001@39in`2000000030;WAk000
0000/<KZ00D0/<KZ0`0000060:^oj0030000002Z_^P0Z[kX0080Z[kX100000190?ooo`030000003o
ool0oooo00l0oooo00<000000?ooo`3oool0103oool002T0oooo00@000000?ooo`3oool000000P2U
`nl300000080YL?_00<000000:G3k`2U`nl01`2U`nl00`000000YL?_0:G3k`0300000080YL?_00<0
00000:G3k`2U`nl01`2U`nl3000000L0YL?_00<000000:G3k`2U`nl01@2U`nl3000000L0YL?_00<0
00000:G3k`2U`nl01@2U`nl3000000P0Y/C_00<000000:S7k`2Xanl0100000020:g<l08000000P2f
f?42000000@0anc^00<000000=Kod@0000001`0000040<WWk0800000102idN`00`000000/<KZ0;36
jP0300000080[<7Y00<000000:^oj02[_nP0102[_nP5000004h0oooo00<000000?ooo`3oool03P3o
ool00`000000oooo0?ooo`040?ooo`00:P3oool2000000030?ooo`000000000000H0YL?_0P000007
0:G3k`<000001P2U`nl2000000D0YL?_0P0000020:G3k`030000002U`nl0YL?_00H0YL?_00<00000
0:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl010000002
0:K3k`030000002Va>l0Y/C_00H0Y/C_0`0000080:g<l08000001@37k>h00`000000e_oA0=Kod@06
00000080fOkE100000060;WAk0<000001P2/`NT00`000000ZkoX0:^oj003000005@0oooo00<00000
0?ooo`3oool03@3oool00`000000oooo0?ooo`040?ooo`00;03oool00`000000oooo000000080:G3
k`030000002U`nl0YL?_00@000000P2U`nl00`000000YL?_0:G3k`060:G3k`030000002U`nl0YL?_
008000001@2U`nl00`000000YL?_0:G3k`050:G3k`<000001`2U`nl00`000000YL?_0:G3k`050:G3
k`<000001`2V`nl00`000000Y/C_0:K4k`030:K4k`8000000P2Xanl00`000000[Lc`0:g<l0050:g<
l080000000<0]]Sa0000000000000P37k>h00`000000e_oA00000004000000030<?[S0000000fOkE
0080fOkE0P0000000`39in`00000000000020;WAk0<000000P2`a^X00`000000[<7Y0:c1j@040:c1
j@<00000F03oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00@0oooo000]0?ooo`04
0000003oool00000000000H0YL?_0`0000070:G3k`8000001P2U`nl3000000P0YL?_00<000000:G3
k`2U`nl0100000020:G3k`030000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl0100000020:G3
k`030000002V`nl0Y/?_00H0Y/?_00<000000:K4k`2Va>l00P0000050:S7k`030000002]c?00[Lc`
0080[Lc`0P0000050;KHl@X000000`3Io]D00`000000bNO/0<WWk0030<WWk08000001P2`a^X00`00
0000[<7Y0:c1j@03000005/0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`040?oo
o`00;P3oool010000000oooo0?ooo`0000020:G3k`<000000P2U`nl00`000000YL?_0:G3k`070:G3
k`030000002U`nl0YL?_00<000000P2U`nl00`000000YL?_0:G3k`070:G3k`8000001`2U`nl20000
00H0YL?_0`0000070:G3k`030000002V`nl0Y/?_00D0Y/?_0`0000080:S7k`040000002]c?000000
000000H0]]Sa0P0000020<O/kP030000003Fom40000000<000000P3Io]D2000000@0bNO/0P000000
0`2idN`000000;36jP050;36jP<00000G`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo00<0oooo000_0?ooo`80000000<0oooo0000000000001P2U`nl2000000L0YL?_0`0000060:G3
k`8000001@2U`nl2000000030:G3k`000000YL?_00P0YL?_00<000000:G3k`2U`nl00`0000020:G3
k`030000002U`nl0YL?_00H0YL?_00<000000:K3k`2V`nl0100000020:K4k`030000002Xanl0Z<O_
00H0Z<O_0P0000060;KHl@8000000`37k>h00`000000e_oA00000003000000040=Wne@0000000000
000000<0bNO/0P0000040;WAk0030000002`a^X0/<KZ00<00000H`3oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo00<0oooo000a0?ooo`030000003oool0000000P0YL?_00<000000:G3
k`2U`nl0100000020:G3k`030000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl00P0000040:G3
k`8000001`2U`nl3000000H0YL?_0P0000060:G3k`<000001`2Va>l200000080Z<O_100000020:g<
l08000000P2ff?42000000@0anc^00<000000=Kod@0000001P0000020=Wne@<000001P2idN`30000
06H0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`030?ooo`00<P3oool00`000000
oooo000000070:G3k`<000001`2U`nl00`000000YL?_0:G3k`050:G3k`<00000202U`nl00`000000
YL?_0:G3k`0400000080YL?_00<000000:G3k`2U`nl01P2U`nl00`000000YL?_0:G3k`0300000080
Y/?_00<000000:K4k`2Va>l01`2Va>l2000000P0[Lc`0P0000050<O/kP030000003Fom40e_oA00<0
00000P33jh`00`000000fOkE0=Wne@02000000030<WWk000000000000080^M7/0`00001Z0?ooo`03
0000003oool0oooo00X0oooo00<000000?ooo`3oool00`3oool003<0oooo00<000000?ooo`000000
0P2U`nl400000080YL?_00<000000:G3k`2U`nl01P2U`nl00`000000YL?_0:G3k`0400000080YL?_
00<000000:G3k`2U`nl01P2U`nl3000000L0YL?_00<000000:G3k`2U`nl01@2U`nl3000000H0Y/?_
0P0000050:K4k`80000000<0Z<O_0000002]c?001P2]c?02000000030;KHl@00000000000080anc^
00<000000=Kod@000000100000001@33jh`000000=Wne@3Io]D0000000D0bNO/0P00001^0?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool00`3oool003@0oooo100000070:G3k`800000
1P2U`nl3000000L0YL?_0P0000020:G3k`@000000P2U`nl00`000000YL?_0:G3k`060:G3k`030000
002U`nl0YL?_00@000000P2U`nl00`000000Y/?_0:K3k`060:K3k`030000002Va>l0Y/C_00800000
102Xanl2000000@0[Lc`00<000000;KHl@2ff?400`2ff?48000000030<?[S0000000fOkE00800000
1039in`200000740oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`030?ooo`00=P3o
ool2000000T0YL?_00<000000:G3k`2U`nl00`0000020:G3k`8000001`2U`nl3000000L0YL?_00<0
00000:G3k`2U`nl01@2U`nl3000000L0YL?_00<000000:K3k`2V`nl01@2V`nl3000000P0Z<O_00@0
00000:g<l000000000001@2ff?4200000080anc^00D000000=Kod@00000000000<?[S004000000<0
bNO/0P00001c0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool00`3oool003L0oooo
0`0000070:G3k`<000001`2U`nl200000080YL?_0`0000020:G3k`030000002U`nl0YL?_00H0YL?_
00<000000:G3k`2U`nl0100000020:G3k`030000002U`nl0YL?_00H0YL?_00<000000:K3k`2V`nl0
100000020:K4k`030000002Xanl0Z<O_00H0Z<O_0P0000060;KHl@0600000037k>h0anc^0<O/kP00
0000e_oA1P0000020=Wne@<00000MP3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo
0080oooo000h0?ooo`030000003oool000000080YL?_100000020:G3k`030000002U`nl0YL?_00H0
YL?_0`0000060:G3k`8000001P2U`nl3000000L0YL?_00<000000:G3k`2U`nl01@2U`nl3000000L0
Y/C_0P0000020:S7k`@000000P2]c?0200000080]]Sa0P0000030<O/kP030000003Fom40e_oA00<0
000000<0`n^<0000003Io]D00P3Io]D00`000000oooo0?ooo`1g0?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool00P3oool003T0oooo100000070:G3k`030000002U`nl0YL?_00<0YL?_
0P0000020:G3k`030000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl00`0000020:G3k`030000
002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl0100000020:K3k`030000002Va>l0Y/C_00H0Y/C_
0`0000080:g<l0<000000`37k>h00`000000e_oA0000000500000080fOkE0P00001j0?ooo`030000
003oool0oooo00L0oooo00<000000?ooo`3oool00P3oool003/0oooo0P0000080:G3k`030000002U
`nl0YL?_008000001@2U`nl00`000000YL?_0:G3k`050:G3k`<000001P2U`nl2000000H0YL?_0`00
00070:K3k`8000000P2Va>l400000080Z<O_00<000000:g<l02]c?00102]c?02000000<0]]Sa2P00
00020=Wne@030000003oool0oooo07/0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?oo
o`020?ooo`00?03oool3000000H0YL?_0`0000080:G3k`030000002U`nl0YL?_00@000000P2U`nl0
0`000000YL?_0:G3k`060:G3k`030000002U`nl0YL?_00<000000P2U`nl00`000000Y/?_0:K3k`06
0:K3k`<000001`2Xanl200000080[Lc`0P0000050;KHl@8000000P37k>h5000000030=Wne@000000
000007l0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`00?@3oool00`00
0000oooo000000020:G3k`<000000P2U`nl2000000L0YL?_0`0000070:G3k`030000002U`nl0YL?_
00D0YL?_0`0000060:G3k`030000002V`nl0Y/?_00<0Y/?_0P0000020:K4k`030000002Xanl0Z<O_
00H0Z<O_0`0000060;KHl@0600000037k>h0anc^0<O/kP000000e_oA1P0000210?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool00P3oool003h0oooo00@000000?ooo`00000000001`2U
`nl200000080YL?_0`0000020:G3k`030000002U`nl0YL?_00H0YL?_00<000000:G3k`2U`nl01000
00020:G3k`030000002U`nl0YL?_00D0YL?_00<000000:K3k`2V`nl00P0000050:K4k`030000002X
anl0Z<O_00<0Z<O_0P0000020:g<l0030000002ff?40]]Sa00<0]]Sa00H000000<O/kP37k>h0anc^
0000003Fom4300000080`n^<00<000000?ooo`3oool0P03oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo0080oooo000o0?ooo`<00000202U`nl3000000H0YL?_0P0000060:G3k`<00000
1`2U`nl00`000000YL?_0:G3k`040:G3k`<00000202Va>l00`000000Z<O_0:S7k`02000000D0[Lc`
0P0000000`2ff?400000000000030<O/kP030000003Fom40e_oA00<0000000<0`n^<0000003oool0
P`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0040oooo00110?ooo`<000000P2U
`nl400000080YL?_00<000000:G3k`2U`nl01P2U`nl00`000000YL?_0:G3k`0300000080YL?_00<0
00000:G3k`2U`nl01P2U`nl00`000000YL?_0:G3k`0300000080Y/?_0P0000070:K4k`<00000202]
c?03000000<0anc^00<000000=Kod@0000000`0000000`33jh`000000?ooo`250?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool00@3oool00480oooo00@000000?ooo`00000000001`2U
`nl00`000000YL?_0:G3k`050:G3k`<000001P2U`nl00`000000YL?_0:G3k`050:G3k`<000001`2V
`nl200000080Y/C_0`0000020:S7k`030000002]c?00[Lc`00@0[Lc`0P0000030;KHl@P0000000<0
`n^<0000003oool0Q`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0040oooo0013
0?ooo`030000003oool0000000P0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002U`nl0
YL?_00D0YL?_00<000000:G3k`2U`nl0100000020:G3k`030000002V`nl0Y/?_00H0Y/?_0`000006
0:S7k`8000000P2]c?02000000D0]]Sa0P0000020<O/kPD00000R@3oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo0040oooo00140?ooo`@000001P2U`nl3000000L0YL?_00<000000:G3
k`2U`nl0102U`nl3000000L0YL?_0P0000020:K3k`@000000P2Va>l00`000000Z<O_0:S7k`060:S7
k`8000001@2ff?42000000<0anc^1@00002;0?ooo`050000003oool0oooo0?ooo`0000000`3oool0
04H0oooo00<000000?ooo`0000000P2U`nl300000080YL?_00<000000:G3k`2U`nl01P2U`nl00`00
0000YL?_0:G3k`0300000080YL?_0P0000070:G3k`<000001`2Va>l00`000000Z<O_0:S7k`030:S7
k`80000000<0[Lc`0000000000000P2ff?42000000@0anc^00<000000=Kod@0000000`00002<0?oo
o`040000003oool0oooo000000<0oooo00170?ooo`040000003oool00000000000H0YL?_0P000004
0:G3k`D000001`2U`nl200000080YL?_0`0000020:K3k`030000002Va>l0Y/C_00H0Y/C_00<00000
0:S7k`2Xanl00P0000050:g<l08000001@37k>h00`000000e_oA00000003000008d0oooo00@00000
0?ooo`3oool000000`3oool004P0oooo00<000000?ooo`0000001`2U`nl5000000@0YL?_00<00000
0:G3k`2U`nl01P2U`nl3000000H0Y/?_00<000000:K4k`2Va>l01@2Va>l3000000H0[Lc`0P000000
0`2ff?400000000000020<O/kPH00000S`3oool00`000000oooo000000030?ooo`00B@3oool40000
0080YL?_0`0000020:G3k`030000002U`nl0YL?_00D0YL?_0P0000020:G3k`@000000P2U`nl00`00
0000Y/?_0:K3k`050:K3k`030000002Va>l0Y/C_00@000000P2Xanl00`000000[Lc`0:g<l0030:g<
l0030000002ff?40]]Sa00<0]]Sa10000000103Fom40000000000000002@0?ooo`030000003oool0
00000080oooo001;0?ooo`040000003oool00000000000H0YL?_0P0000060:G3k`<000001`2U`nl0
0`000000Y/?_0:K3k`040:K3k`<000001`2Xanl2000000030:g<l0000000000000D0]]Sa0P000000
0`37k>h000000=Kod@0300000980oooo0P0000020?ooo`00C03oool00`000000oooo000000080:G3
k`030000002U`nl0YL?_00<000000P2U`nl00`000000YL?_0:G3k`060:G3k`030000002V`nl0Y/?_
00<000000P2Va>l00`000000Z<O_0:S7k`060:S7k`8000001P2ff?4010000000anc^0<O/kP37k>h4
000009<0oooo0P0000020?ooo`00C@3oool00`000000oooo000000070:G3k`<000001P2U`nl00`00
0000YL?_0:G3k`050:G3k`<000001P2Va>l200000080Z<O_100000020:g<l08000000P2ff?420000
00<0anc^00<000000=Kod@0000000P00002C0?ooo`8000000P3oool004h0oooo0`0000020:G3k`@0
00000P2U`nl00`000000YL?_0:G3k`050:G3k`030000002U`nl0YL?_00@000000P2V`nl00`000000
Y/C_0:K4k`050:K4k`<00000202]c?03000000<0anc^1@00002@0?ooo`@00000103oool00500oooo
0`0000070:G3k`800000102U`nl4000000L0Y/?_0P0000020:K4k`<000000P2Xanl00`000000[Lc`
0:g<l0040:g<l08000000`2ff?47000008d0oooo100000080?ooo`00D@3oool3000000L0YL?_1@00
00030:G3k`030000002V`nl0Y/?_00H0Y/?_0`0000060:S7k`8000000P2]c?02000000D0]]Sa0P00
00000`37k>h000000=Kod@02000008T0oooo1000000<0?ooo`00DP3oool010000000oooo00000000
00020:G3k`<000000P2U`nl00`000000YL?_0:G3k`040:G3k`8000000P2V`nl400000080Y/C_00<0
00000:S7k`2Xanl01@2Xanl3000000D0]]Sa0P0000020<O/kP040000003Fom400000000008H0oooo
1000000@0?ooo`00D`3oool2000000030?ooo`000000000000H0YL?_00<000000:G3k`2U`nl0102U
`nl3000000L0Y/C_0P0000020:S7k`<000000P2]c?0200000080]]Sa0P0000030<O/kP030000003F
om40e_oA00800000PP3oool4000001@0oooo001E0?ooo`030000003oool0000000L0YL?_00<00000
0:G3k`2U`nl00`0000020:K3k`030000002Va>l0Y/C_00H0Y/C_0`0000070:g<l0<000000`37k>h0
0`000000e_oA0000000200000800oooo0`00000H0?ooo`00EP3oool00`000000oooo000000060:G3
k`<000001P2V`nl200000080Y/C_100000020:S7k`030000002]c?00[Lc`00@0[Lc`0P0000020;KH
l@P00000O03oool4000001/0oooo001G0?ooo`030000003oool000000080YL?_0`0000020:G3k`03
0000002V`nl0Y/?_00D0Y/?_0`0000070:S7k`8000000P2]c?02000000@0]]Sa0P0000020<O/kP<0
0000N@3oool4000001l0oooo001H0?ooo`040000003oool00000000000H0YL?_0P0000020:K3k`<0
00000P2Va>l00`000000Z<O_0:S7k`060:S7k`<000001@2ff?4010000000anc^0<O/kP37k>h40000
07D0oooo1000000S0?ooo`00F@3oool3000000L0YL?_0`0000060:K4k`8000000P2Xanl400000080
[Lc`0P0000020;KHl@8000000`37k>h010000000e_oA00000000001b0?ooo`@000009`3oool005/0
oooo0`0000020:G3k`<000000P2V`nl00`000000Y/C_0:K4k`050:K4k`<00000202]c?03000000<0
anc^1@00001^0?ooo`@00000:`3oool005`0oooo00@000000?ooo`00000000001P2V`nl200000080
Y/C_0`0000020:S7k`030000002]c?00[Lc`00@0[Lc`0P0000030;KHl@L00000K03oool3000002l0
oooo001M0?ooo`030000003oool0000000L0Y/?_0`0000060:S7k`8000000P2]c?02000000D0]]Sa
0P0000000`37k>h000000=Kod@02000006P0oooo1000000b0?ooo`00GP3oool400000080Y/?_0`00
00020:K4k`030000002Xanl0Z<O_00D0Z<O_0`0000050;KHl@8000000P37k>h00`000000e_oA0000
0002000006@0oooo1000000f0?ooo`00H03oool010000000oooo0000000000060:K4k`8000000P2X
anl300000080[Lc`0P0000020;KHl@8000000`37k>h00`000000e_oA0=Kod@0300000600oooo1000
000j0?ooo`00H@3oool00`000000oooo000000070:K4k`<000001`2]c?03000000<0anc^00<00000
0=Kod@0000000P0000000`3oool000000?ooo`1K0?ooo`@00000?P3oool00680oooo00@000000?oo
o`00000000000P2Va>l300000080Z<O_00<000000:g<l02]c?00102]c?0200000080]]Sa20000000
0`3oool000000?ooo`1G0?ooo`@00000@P3oool006<0oooo0P0000000`3oool00000000000060:S7
k`8000000P2]c?02000000@0]]Sa0P0000020<O/kP<000000P3oool00`000000oooo0?ooo`1C0?oo
o`<00000AP3oool006D0oooo00<000000?ooo`0000001`2Xanl3000000D0]]Sa00H000000<O/kP37
k>h0anc^0000003Fom4200000080oooo00<000000?ooo`3oool0C`3oool4000004T0oooo001V0?oo
o`040000003oool0000000000080Z<O_0`0000020:g<l08000000P2ff?42000000<0anc^00@00000
0=Kod@00000000000`3oool00`000000oooo0?ooo`1;0?ooo`@00000C@3oool006L0oooo00<00000
0?ooo`3oool00P0000070:g<l0<000000`37k>h4000000@0oooo00<000000?ooo`3oool0A`3oool4
00000540oooo001X0?ooo`80000000<0oooo0000002]c?001@2]c?0200000080]]Sa1P0000050?oo
o`030000003oool0oooo04<0oooo1000001E0?ooo`00JP3oool010000000oooo0000000000020:g<
l0800000102ff?42000000040<O/kP0000000000000000D0oooo00<000000?ooo`3oool0@03oool3
000005T0oooo001[0?ooo`030000003oool0oooo008000001@2ff?401P000000anc^0<O/kP000000
e_oA000000H0oooo00<000000?ooo`3oool0?03oool4000005`0oooo001/0?ooo`030000003oool0
oooo008000000P2ff?4200000080anc^00<000000=Kod@0000001`3oool00`000000oooo0?ooo`0h
0?ooo`@00000H03oool006d0oooo0P0000020?ooo`<000000P37k>h3000000P0oooo00<000000?oo
o`3oool0=03oool4000006@0oooo001_0?ooo`030000003oool0oooo0080oooo0`00000:0?ooo`03
0000003oool0oooo0300oooo1000001X0?ooo`00L03oool00`000000oooo0?ooo`0=0?ooo`030000
003oool0oooo02d0oooo1000001/0?ooo`00L@3oool00`000000oooo0?ooo`0<0?ooo`030000003o
ool0oooo02X0oooo0`00001`0?ooo`00LP3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo02H0oooo1000001c0?ooo`00L`3oool2000000/0oooo00<000000?ooo`3oool08P3oool40000
07L0oooo001e0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool07P3oool4000007/0
oooo001f0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool06P3oool4000007l0oooo
001g0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool05P3oool4000008<0oooo001h
0?ooo`8000001P3oool00`000000oooo0?ooo`0C0?ooo`<00000Q`3oool007X0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`0?0?ooo`@00000RP3oool007/0oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`0;0?ooo`@00000SP3oool007`0oooo00D000000?ooo`3oool0
oooo000000090?ooo`@00000TP3oool007d0oooo0P0000000`3oool000000?ooo`040?ooo`@00000
UP3oool007l0oooo0P0000000`3oool0000000000002000009X0oooo00200?ooo`800000WP3oool0
0?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {232.563, 0}} -> {-0.0895205, 0.00532715, \
0.00364727, 0.00364727}}]
}, Open  ]],

Cell["\<\
You can see from the plot that with \[Nu] = 0.025, there is a fairly steep \
front. This moves with constant speed.\
\>", "Text"],

Cell[TextData[{
  "This solves the problem using the default settings for ",
  StyleBox["NDSolve", "MR"],
  " and the IDA method with the exception of the ",
  StyleBox["DifferentiateBoundaryConditions", "MR"],
  " option for ",
  StyleBox["MethodOfLines", "MR"],
  ", which causes ",
  StyleBox["NDSolve", "MR"],
  " to treat the boundary conditions as algebraic. "
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Timing[
      sol\  = \ NDSolve[{de, \ ic, \ bc}, 
          u, {t, 0, 1}, {x, 0, 1}, {y, 0, 1}, \ 
          Method \[Rule] {"\<MethodOfLines\>", \ \
"\<DifferentiateBoundaryConditions\>" \[Rule] False}]]\)], "Input",
  CellLabel->"In[54]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(2.083`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{"{", 
            RowBox[{"u", "\[Rule]", 
              
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}, {0.`, 1.`}, {0.`, 
                      1.`}}, "<>"]\),
                False,
                Editable->False]}], "}"}], "}"}]}], "}"}]], "Output",
  CellLabel->"Out[54]="]
}, Open  ]],

Cell["\<\
Since there is an exact solution to compare to, the overall solution accuracy \
can be compared as well.\
\>", "Text"],

Cell["\<\
This defines a function that finds the maximum deviation between the exact \
and computed solutions at the grid points at all of the time steps.\
\>", "MathCaption"],

Cell[BoxData[
    \(errfun[sol_]\  := \ 
      Module[{ifun\  = \ First[u\  /. \ sol], \ grid, \ exvals, \ 
          gvals}, \[IndentingNewLine]grid\  = \ 
          InterpolatingFunctionGrid[ifun]; \[IndentingNewLine]gvals\  = \ 
          InterpolatingFunctionValuesOnGrid[
            ifun]; \[IndentingNewLine]exvals\  = \ 
          Apply[Bsol, 
            Transpose[grid, \ 
              RotateLeft[Range[ArrayDepth[grid]], 
                1]]]; \[IndentingNewLine]Max[
          Abs[exvals\  - \ gvals]]]\)], "Input",
  CellLabel->"In[55]:="],

Cell["This computes the maximal error for the computed solution. ", \
"MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(errfun[sol]\)], "Input",
  CellLabel->"In[56]:="],

Cell[BoxData[
    \(0.0007494462437129257`\)], "Output",
  CellLabel->"Out[56]="]
}, Open  ]],

Cell["\<\
In the following, a comparison will be made with different settings for the \
options of the IDA method. To emphasize the option settings, a function will \
be defined to time the computation of the solution and give the maximal \
error.\
\>", "Text"],

Cell["\<\
This defines a function for comparing different IDA option settings.\
\>", "MathCaption"],

Cell[BoxData[
    \(TimeSolution[idaopts___]\  := \ 
      Module[{time, \ err, \ steps}, \[IndentingNewLine]time\  = \ 
          First[Timing[
              sol\  = \ 
                NDSolve[{de, \ ic, \ bc}, 
                  u, {t, 0, 1}, {x, 0, 1}, {y, 0, 1}, \ 
                  Method \[Rule] {"\<MethodOfLines\>", \ \
"\<DifferentiateBoundaryConditions\>" \[Rule] False, \ 
                      Method \[Rule] {IDA, \ 
                          idaopts}}]]]; \[IndentingNewLine]err\  = \ 
          errfun[sol]; \[IndentingNewLine]steps\  = \ 
          Length[First[
                InterpolatingFunctionCoordinates[
                  First[u\  /. \ 
                      sol]]]]\ "\<Steps\>"; \[IndentingNewLine]{time, \ 
          err, \ steps}]\)], "Input",
  CellLabel->"In[57]:="],

Cell["No options use the previous defaults.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TimeSolution[]\)], "Input",
  CellLabel->"In[58]:="],

Cell[BoxData[
    \({2.0329999999999995`\ Second, 0.0007494462437129257`, 
      88\ "Steps"}\)], "Output",
  CellLabel->"Out[58]="]
}, Open  ]],

Cell[TextData[{
  "This uses the ",
  StyleBox["Band", "MR"],
  " method."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TimeSolution["\<ImplicitSolver\>" \[Rule] {"\<Newton\>", \ \
"\<LinearSolveMethod\>" \[Rule] "\<Band\>"}]\)], "Input",
  CellLabel->"In[59]:="],

Cell[BoxData[
    \({8.202`\ Second, 0.0007494967868604679`, 88\ "Steps"}\)], "Output",
  CellLabel->"Out[59]="]
}, Open  ]],

Cell[TextData[{
  "The ",
  StyleBox["Band", "MR"],
  " method is not very effective because the bandwidth of the Jacobian is \
relatively large, partly because of the fourth order derivatives used and \
partly because the one-sided stencils used near the boundary add width at the \
top and bottom. You can specify the bandwidth explicitly."
}], "Text"],

Cell[TextData[{
  "This uses the ",
  StyleBox["Band", "MR"],
  " method with the width set to include the stencil of the differences in \
only one direction."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TimeSolution["\<ImplicitSolver\>" \[Rule] {"\<Newton\>", \ \
"\<LinearSolveMethod\>" \[Rule] {"\<Band\>", \ "\<BandWidth\>" \[Rule] 
                3}}]\)], "Input",
  CellLabel->"In[60]:="],

Cell[BoxData[
    \({6.920000000000002`\ Second, 0.000937962328172337`, 
      311\ "Steps"}\)], "Output",
  CellLabel->"Out[60]="]
}, Open  ]],

Cell["\<\
While the solution time was smaller, notice that the error is slightly \
greater and the total number of time steps is a lot greater. If the problem \
was more stiff, the iterations likely would not have converged because it was \
missing information from the other direction. Ideally, the bandwidth should \
not eliminate information from an entire dimension. \
\>", "Text"],

Cell[TextData[{
  "This computes the grids used in the ",
  StyleBox["X",
    FontSlant->"Italic"],
  " and ",
  StyleBox["Y",
    FontSlant->"Italic"],
  " directions and shows the number of points used."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \({X, 
        Y}\  = \ \(InterpolatingFunctionCoordinates[
          First[u\  /. \ sol]]\)[\([\)\({2, 
          3}\)\(]\)]; \), "\[IndentingNewLine]", 
    \({nx, ny}\  = \ {Length[X], \ Length[Y]}\)}], "Input",
  CellLabel->"In[61]:="],

Cell[BoxData[
    \({51, 51}\)], "Output",
  CellLabel->"Out[62]="]
}, Open  ]],

Cell[TextData[{
  "This uses the ",
  StyleBox["Band", "MR"],
  " method with the width set to include at least part of the stencil in both \
directions."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TimeSolution["\<ImplicitSolver\>" \[Rule] {"\<Newton\>", \ \
"\<LinearSolveMethod\>" \[Rule] {"\<Band\>", \ "\<BandWidth\>" \[Rule] 
                51}}]\)], "Input",
  CellLabel->"In[63]:="],

Cell[BoxData[
    \({2.1930000000000014`\ Second, 0.0008597296528987108`, 
      88\ "Steps"}\)], "Output",
  CellLabel->"Out[63]="]
}, Open  ]],

Cell[TextData[{
  "With the more appropriate setting of the bandwidth, the solution is still \
slightly slower than in the default case. The ",
  StyleBox["Band", "MR"],
  " method can sometimes be effective on two-dimensional problems, but is \
usually most effective on one-dimensional problems."
}], "Text"],

Cell[TextData[{
  "This computes the solution using the ",
  StyleBox["GMRES", "MR"],
  " implicit solver without a preconditioner."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TimeSolution["\<ImplicitSolver\>" \[Rule] "\<GMRES\>"]\)], "Input",
  CellLabel->"In[64]:="],

Cell[BoxData[
    \({25.107000000000003`\ Second, 0.004354419844594948`, 
      661\ "Steps"}\)], "Output",
  CellLabel->"Out[64]="]
}, Open  ]],

Cell[TextData[{
  "This is incredibly slow! Using the ",
  StyleBox["GMRES", "MR"],
  " method without a preconditioner is not recommended for this very reason. \
However, finding a good preconditioner is not usually trivial. For this \
example, a diagonal preconditioner will be used."
}], "Text"],

Cell[TextData[{
  "The setting of the ",
  StyleBox["Preconditioner", "MR"],
  " option should be a function ",
  StyleBox["f", "MR"],
  ", which accepts four arguments that will be given to it by ",
  StyleBox["NDSolve", "MR"],
  " such that ",
  StyleBox["f[t, x, x', c]", "MR"],
  " returns another function that will apply the preconditioner to the \
residual vector. (See IDA user guide [",
  ButtonBox["HT99",
    ButtonData:>{"References.nb", "HT99"},
    ButtonStyle->"Hyperlink"],
  "] for details on how the preconditioner is used.) The arguments ",
  StyleBox["t, x, x', c", "MR"],
  " are the current time, solution vector, solution derivative vector, and \
the constant ",
  StyleBox["c",
    FontSlant->"Italic"],
  " in formula (2) above. For example, if you can determine a procedure that \
would generate an appropriate preconditioner matrix ",
  StyleBox["P",
    FontSlant->"Italic"],
  " as a function of these arguments, you could use\n",
  StyleBox["Preconditioner->Function[{t,x,xp,c}, LinearSolve[P[t,x,xp,c]]]", 
    "MR"],
  "\nto produce a ",
  StyleBox["LinearSolveFunction", "MR"],
  " object which will effectively invert the preconditioner matrix ",
  StyleBox["P",
    FontSlant->"Italic"],
  ". Typically, for each time the preconditioner function is set up, it is \
applied to the residual vector several times, so using some sort of \
factorization such as is contained in a ",
  StyleBox["LinearSolveFunction", "MR"],
  " is a good idea."
}], "Text"],

Cell["\<\
For the diagonal case, the inverse can be effected simply by using the \
reciprocal. The most difficult part of setting up a diagonal preconditioner \
is keeping in mind that values on the boundary should not be affected by it. \
\
\>", "Text"],

Cell["\<\
This finds the diagonal elements of the differentiation matrix for computing \
the preconditioner.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(DM\  = \ 
      NDSolve`FiniteDifferenceDerivative[{2, 0}, {X, 
              Y}]@"\<DifferentiationMatrix\>"[]\  + \ 
        NDSolve`FiniteDifferenceDerivative[{0, 2}, {X, 
              Y}]@"\<DifferentiationMatrix\>"[]; \), "\[IndentingNewLine]", 
    \(Short[diag\  = \ Tr[DM, List]]\)}], "Input",
  CellLabel->"In[65]:="],

Cell[BoxData[
    TagBox[\({18750.`, 6250.`, 3125.`, \[LeftSkeleton]2595\[RightSkeleton], 
        3125.`, 6250.000000000001`, 18750.`}\),
      Short]], "Output",
  CellLabel->"Out[66]//Short="]
}, Open  ]],

Cell["\<\
This gets the positions where elements at the boundary that satisfy a simple \
algebraic condition are in the flattened solution vector.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(bound\  = \ 
      SparseArray[{{i_, \ 1} \[Rule] 1. , \ {i_, \ ny} \[Rule] 
            1. , {1, i_} \[Rule] 1. , {nx, i_} \[Rule] 1. }, {nx, ny}, 
        0. ]; \), "\[IndentingNewLine]", 
    \(Short[
      pos\  = \ \(Drop[ArrayRules[Flatten[bound]], \(-1\)]\)[\([\)\(All, 1, 
          1\)\(]\)]]\)}], "Input",
  CellLabel->"In[67]:="],

Cell[BoxData[
    TagBox[\({1, 2, 3, 4, 5, 6, 7, 8, 9, \[LeftSkeleton]182\[RightSkeleton], 
        2593, 2594, 2595, 2596, 2597, 2598, 2599, 2600, 2601}\),
      Short]], "Output",
  CellLabel->"Out[68]//Short="]
}, Open  ]],

Cell["\<\
This defines the function that sets up the function called to get the \
effective inverse of the preconditioner. For the diagonal case, the inverse \
is done simply by taking the reciprocal.\
\>", "MathCaption"],

Cell[BoxData[
    \(pfun[t_, \ x_, \ xp_, \ c_]\  := \ 
      Module[{d, dd}, \[IndentingNewLine]d\  = \ 
          1. /\((c\  - \ \[Nu]\ diag)\); \[IndentingNewLine]d[\([\)\(pos\)\(]\
\)]\  = \ 1. ; \[IndentingNewLine]Function[#\ dd]\  /. \ 
          dd \[Rule] d]\)], "Input",
  CellLabel->"In[69]:="],

Cell[TextData[{
  "This uses the preconditioned ",
  StyleBox["GMRES", "MR"],
  " method to compute the solution."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TimeSolution["\<ImplicitSolver\>" \[Rule] {"\<GMRES\>", \ \
"\<Preconditioner\>" \[Rule] pfun}]\)], "Input",
  CellLabel->"In[70]:="],

Cell[BoxData[
    \({1.0720000000000027`\ Second, 0.0007160064681386302`, 
      88\ "Steps"}\)], "Output",
  CellLabel->"Out[70]="]
}, Open  ]],

Cell[TextData[{
  "Thus, even with a crude preconditioner, the ",
  StyleBox["GMRES", "MR"],
  " method computes the solution faster than the using the direct sparse \
solvers. "
}], "Text"],

Cell[TextData[{
  "For PDE discretizations with higher-order temporal derivatives or systems \
of PDEs, you may need to look at the corresponding ",
  StyleBox["NDSolve`StateData", "MR"],
  " object to determine how the variables are ordered so that you can get the \
structural form of the preconditioner correctly."
}], "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.1 for Microsoft Windows",
ScreenRectangle->{{0, 1400}, {0, 967}},
AutoGeneratedPackage->None,
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{650, 500},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Differential-Algebraic Equations",
Magnification->1,
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "c:1"->{
    Cell[1852, 57, 72, 1, 65, "Section",
      CellTags->{"c:1", "b:10", "ndsg:4.0"}]},
  "b:10"->{
    Cell[1852, 57, 72, 1, 65, "Section",
      CellTags->{"c:1", "b:10", "ndsg:4.0"}]},
  "ndsg:4.0"->{
    Cell[1852, 57, 72, 1, 65, "Section",
      CellTags->{"c:1", "b:10", "ndsg:4.0"}]},
  "c:2"->{
    Cell[467591, 17810, 45, 1, 65, "Section",
      CellTags->"c:2"]},
  "c:3"->{
    Cell[467661, 17815, 77, 1, 39, "Subsection",
      CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"}]},
  "b:11.0"->{
    Cell[467661, 17815, 77, 1, 39, "Subsection",
      CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"}]},
  "ndsg:4.1.0"->{
    Cell[467661, 17815, 77, 1, 39, "Subsection",
      CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"}]},
  "IDA"->{
    Cell[467661, 17815, 77, 1, 39, "Subsection",
      CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"}]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"c:1", 767477, 26140},
  {"b:10", 767577, 26143},
  {"ndsg:4.0", 767681, 26146},
  {"c:2", 767780, 26149},
  {"c:3", 767862, 26152},
  {"b:11.0", 767983, 26155},
  {"ndsg:4.1.0", 768108, 26158},
  {"IDA", 768226, 26161}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 51, 0, 87, "Chapter"],

Cell[CellGroupData[{
Cell[1852, 57, 72, 1, 65, "Section",
  CellTags->{"c:1", "b:10", "ndsg:4.0"}],
Cell[1927, 60, 125, 3, 25, "Text"],
Cell[2055, 65, 262, 8, 31, "DisplayMath"],
Cell[2320, 75, 589, 16, 59, "Text"],
Cell[2912, 93, 273, 5, 42, "Text"],
Cell[3188, 100, 258, 7, 31, "NumberedEquation"],
Cell[3449, 109, 353, 12, 25, "Text"],
Cell[3805, 123, 581, 13, 76, "Text"],
Cell[4389, 138, 277, 6, 42, "Text"],
Cell[4669, 146, 144, 3, 42, "Text"],
Cell[4816, 151, 112, 3, 41, "MathCaption"],
Cell[4931, 156, 231, 5, 69, "Input",
  InitializationCell->True],
Cell[5165, 163, 520, 11, 76, "Text"],
Cell[5688, 176, 111, 3, 41, "MathCaption"],
Cell[5802, 181, 525, 13, 56, "Input"],
Cell[6330, 196, 197, 7, 25, "Text"],
Cell[6530, 205, 203, 7, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[6758, 216, 159, 4, 29, "Input"],
Cell[6920, 222, 614, 16, 65, "Output"]
}, Open  ]],
Cell[7549, 241, 366, 10, 59, "Text"],
Cell[7918, 253, 119, 4, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[8062, 261, 129, 3, 29, "Input"],
Cell[8194, 266, 82, 2, 27, "Output"]
}, Open  ]],
Cell[8291, 271, 270, 8, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[8586, 283, 285, 6, 49, "Input"],
Cell[8874, 291, 16782, 443, 184, 4099, 281, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[25671, 737, 119, 3, 25, "Text"],
Cell[25793, 742, 169, 6, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[25987, 752, 162, 4, 29, "Input"],
Cell[26152, 758, 171, 4, 20, "Message"],
Cell[26326, 764, 614, 16, 65, "Output"]
}, Open  ]],
Cell[26955, 783, 119, 4, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[27099, 791, 130, 3, 29, "Input"],
Cell[27232, 796, 82, 2, 27, "Output"]
}, Open  ]],
Cell[27329, 801, 222, 8, 25, "Text"],
Cell[27554, 811, 119, 5, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[27698, 820, 85, 2, 29, "Input"],
Cell[27786, 824, 366, 10, 27, "Output"]
}, Open  ]],
Cell[28167, 837, 408, 12, 42, "Text"],
Cell[28578, 851, 340, 14, 42, "Text"],
Cell[28921, 867, 208, 7, 42, "Text"],
Cell[29132, 876, 243, 8, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[29400, 888, 184, 4, 29, "Input"],
Cell[29587, 894, 615, 16, 65, "Output"]
}, Open  ]],
Cell[30217, 913, 92, 2, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[30334, 919, 240, 5, 49, "Input"],
Cell[30577, 926, 16720, 461, 184, 4404, 304, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[47312, 1390, 555, 19, 59, "Text"],
Cell[47870, 1411, 497, 10, 59, "Text"],
Cell[48370, 1423, 114, 3, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[48509, 1430, 142, 3, 29, "Input"],
Cell[48654, 1435, 291, 5, 65, "Message"],
Cell[48948, 1442, 61, 2, 27, "Output"]
}, Open  ]],
Cell[49024, 1447, 493, 11, 76, "Text"],
Cell[49520, 1460, 137, 3, 25, "Text"],
Cell[49660, 1465, 156, 3, 58, "MathCaption"],
Cell[49819, 1470, 542, 11, 103, "Input"],
Cell[50364, 1483, 126, 3, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[50515, 1490, 190, 4, 49, "Input"],
Cell[50708, 1496, 574, 15, 65, "Output"]
}, Open  ]],
Cell[51297, 1514, 264, 8, 42, "Text"],
Cell[51564, 1524, 150, 8, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[51739, 1536, 379, 8, 109, "Input"],
Cell[52121, 1546, 36560, 975, 243, 9275, 633, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[88696, 2524, 75, 0, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[88796, 2528, 184, 4, 49, "Input"],
Cell[88983, 2534, 574, 15, 65, "Output"]
}, Open  ]],
Cell[89572, 2552, 165, 3, 42, "Text"],
Cell[89740, 2557, 303, 7, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[90068, 2568, 794, 18, 169, "Input"],
Cell[90865, 2588, 53094, 1814, 184, 34876, 1584, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[143974, 4405, 361, 6, 59, "Text"],
Cell[144338, 4413, 59, 0, 25, "Text"],
Cell[144400, 4415, 380, 10, 47, "DisplayMath"],
Cell[144783, 4427, 498, 12, 76, "Text"],
Cell[145284, 4441, 138, 3, 25, "Text"],
Cell[145425, 4446, 150, 3, 41, "MathCaption"],
Cell[145578, 4451, 325, 7, 69, "Input"],
Cell[145906, 4460, 167, 3, 42, "Text"],
Cell[146076, 4465, 351, 10, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[146452, 4479, 197, 4, 49, "Input"],
Cell[146652, 4485, 217, 6, 27, "Output"]
}, Open  ]],
Cell[146884, 4494, 610, 20, 59, "Text"],
Cell[147497, 4516, 246, 5, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[147768, 4525, 174, 4, 49, "Input"],
Cell[147945, 4531, 217, 6, 27, "Output"]
}, Open  ]],
Cell[148177, 4540, 93, 2, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[148295, 4546, 103, 2, 29, "Input"],
Cell[148401, 4550, 75701, 1387, 184, 8237, 549, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[224117, 5940, 145, 4, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[224287, 5948, 308, 7, 69, "Input"],
Cell[224598, 5957, 121393, 5939, 184, 107140, 5758, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[346006, 11899, 193, 4, 42, "Text"],
Cell[346202, 11905, 138, 3, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[346365, 11912, 258, 5, 49, "Input"],
Cell[346626, 11919, 217, 6, 27, "Output"]
}, Open  ]],
Cell[346858, 11928, 172, 4, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[347055, 11936, 310, 7, 69, "Input"],
Cell[347368, 11945, 120174, 5859, 184, 106613, 5686, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[467591, 17810, 45, 1, 65, "Section",
  CellTags->"c:2"],

Cell[CellGroupData[{
Cell[467661, 17815, 77, 1, 39, "Subsection",
  CellTags->{"c:3", "b:11.0", "ndsg:4.1.0", "IDA"}],
Cell[467741, 17818, 1218, 32, 93, "Text"],
Cell[468962, 17852, 448, 13, 59, "Text"],
Cell[469413, 17867, 363, 10, 42, "Text"],
Cell[469779, 17879, 295, 7, 56, "DisplayMath"],
Cell[470077, 17888, 300, 9, 42, "Text"],
Cell[470380, 17899, 158, 3, 56, "DisplayMath"],
Cell[470541, 17904, 137, 3, 25, "Text"],
Cell[470681, 17909, 579, 15, 50, "NumberedEquation"],
Cell[471263, 17926, 699, 18, 76, "Text"],
Cell[471965, 17946, 896, 32, 42, "Text"],
Cell[472864, 17980, 189, 3, 52, "DisplayMath"],
Cell[473056, 17985, 305, 13, 42, "Text"],
Cell[473364, 18000, 452, 12, 60, "Text"],
Cell[473819, 18014, 213, 8, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[474057, 18026, 78, 2, 29, "Input"],
Cell[474138, 18030, 162, 3, 27, "Output"]
}, Open  ]],
Cell[474315, 18036, 921, 17, 152, "DefinitionBox3Col"],
Cell[475239, 18055, 76, 3, 17, "Caption"],
Cell[475318, 18060, 637, 12, 93, "Text"],
Cell[475958, 18074, 370, 9, 59, "Text"],
Cell[476331, 18085, 136, 3, 41, "MathCaption"],
Cell[476470, 18090, 93, 2, 29, "Input"],
Cell[476566, 18094, 95, 2, 41, "MathCaption"],
Cell[476664, 18098, 251, 5, 49, "Input"],
Cell[476918, 18105, 154, 5, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[477097, 18114, 131, 3, 29, "Input"],
Cell[477231, 18119, 212, 4, 40, "Output"]
}, Open  ]],
Cell[477458, 18126, 116, 3, 25, "Text"],
Cell[477577, 18131, 166, 3, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[477768, 18138, 151, 4, 29, "Input"],
Cell[477922, 18144, 420, 12, 46, "Output"]
}, Open  ]],
Cell[478357, 18159, 65, 0, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[478447, 18163, 256, 6, 49, "Input"],
Cell[478706, 18171, 420, 12, 46, "Output"]
}, Open  ]],
Cell[479141, 18186, 252, 4, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[479418, 18194, 684, 13, 129, "Input"],
Cell[480105, 18209, 56606, 1575, 184, 16972, 1081, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[536726, 19787, 353, 6, 59, "Text"],
Cell[537082, 19795, 97, 2, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[537204, 19801, 98, 2, 29, "Input"],
Cell[537305, 19805, 72, 2, 27, "Output"]
}, Open  ]],
Cell[537392, 19810, 243, 4, 42, "Text"],
Cell[537638, 19816, 158, 3, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[537821, 19823, 195, 5, 49, "Input"],
Cell[538019, 19830, 180, 5, 27, "Output"]
}, Open  ]],
Cell[538214, 19838, 72, 0, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[538311, 19842, 285, 6, 69, "Input"],
Cell[538599, 19850, 180, 5, 27, "Output"]
}, Open  ]],
Cell[538794, 19858, 237, 4, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[539056, 19866, 675, 12, 149, "Input"],
Cell[539734, 19880, 88073, 3207, 184, 39974, 2608, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[627822, 23090, 155, 3, 42, "Text"],
Cell[627980, 23095, 607, 15, 76, "Text"],
Cell[628590, 23112, 353, 10, 42, "Text"],
Cell[628946, 23124, 987, 23, 198, "DefinitionBox"],
Cell[629936, 23149, 115, 4, 17, "Caption"],
Cell[630054, 23155, 501, 12, 59, "Text"],
Cell[630558, 23169, 1579, 35, 216, "DefinitionBox3Col"],
Cell[632140, 23206, 79, 3, 17, "Caption"],
Cell[632222, 23211, 112, 3, 25, "Text"],
Cell[632337, 23216, 153, 3, 47, "DisplayMath"],
Cell[632493, 23221, 453, 8, 76, "Text"],
Cell[632949, 23231, 105, 2, 41, "MathCaption"],
Cell[633057, 23235, 141, 3, 29, "Input"],
Cell[633201, 23240, 133, 3, 41, "MathCaption"],
Cell[633337, 23245, 398, 8, 89, "Input"],
Cell[633738, 23255, 62, 0, 41, "MathCaption"],
Cell[633803, 23257, 278, 7, 49, "Input"],
Cell[634084, 23266, 247, 7, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[634356, 23277, 158, 3, 49, "Input"],
Cell[634517, 23282, 119599, 2431, 240, 52186, 1594, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[754131, 25716, 139, 3, 25, "Text"],
Cell[754273, 25721, 385, 10, 75, "MathCaption"],

Cell[CellGroupData[{
Cell[754683, 25735, 263, 6, 49, "Input"],
Cell[754949, 25743, 402, 11, 46, "Output"]
}, Open  ]],
Cell[755366, 25757, 128, 3, 25, "Text"],
Cell[755497, 25762, 175, 3, 58, "MathCaption"],
Cell[755675, 25767, 553, 12, 129, "Input"],
Cell[756231, 25781, 84, 1, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[756340, 25786, 69, 2, 29, "Input"],
Cell[756412, 25790, 81, 2, 27, "Output"]
}, Open  ]],
Cell[756508, 25795, 261, 5, 59, "Text"],
Cell[756772, 25802, 99, 2, 41, "MathCaption"],
Cell[756874, 25806, 799, 17, 189, "Input"],
Cell[757676, 25825, 60, 0, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[757761, 25829, 72, 2, 29, "Input"],
Cell[757836, 25833, 132, 3, 27, "Output"]
}, Open  ]],
Cell[757983, 25839, 93, 4, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[758101, 25847, 163, 3, 29, "Input"],
Cell[758267, 25852, 112, 2, 27, "Output"]
}, Open  ]],
Cell[758394, 25857, 354, 7, 59, "Text"],
Cell[758751, 25866, 178, 5, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[758954, 25875, 211, 4, 49, "Input"],
Cell[759168, 25881, 131, 3, 27, "Output"]
}, Open  ]],
Cell[759314, 25887, 385, 6, 76, "Text"],
Cell[759702, 25895, 224, 8, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[759951, 25907, 258, 6, 49, "Input"],
Cell[760212, 25915, 67, 2, 27, "Output"]
}, Open  ]],
Cell[760294, 25920, 173, 5, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[760492, 25929, 212, 4, 49, "Input"],
Cell[760707, 25935, 132, 3, 27, "Output"]
}, Open  ]],
Cell[760854, 25941, 310, 6, 59, "Text"],
Cell[761167, 25949, 151, 4, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[761343, 25957, 112, 2, 29, "Input"],
Cell[761458, 25961, 132, 3, 27, "Output"]
}, Open  ]],
Cell[761605, 25967, 298, 6, 59, "Text"],
Cell[761906, 25975, 1486, 37, 194, "Text"],
Cell[763395, 26014, 254, 5, 42, "Text"],
Cell[763652, 26021, 129, 3, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[763806, 26028, 348, 7, 109, "Input"],
Cell[764157, 26037, 195, 4, 27, "Output"]
}, Open  ]],
Cell[764367, 26044, 167, 3, 58, "MathCaption"],

Cell[CellGroupData[{
Cell[764559, 26051, 361, 8, 69, "Input"],
Cell[764923, 26061, 213, 4, 46, "Output"]
}, Open  ]],
Cell[765151, 26068, 221, 4, 58, "MathCaption"],
Cell[765375, 26074, 304, 6, 89, "Input"],
Cell[765682, 26082, 133, 4, 41, "MathCaption"],

Cell[CellGroupData[{
Cell[765840, 26090, 153, 3, 29, "Input"],
Cell[765996, 26095, 132, 3, 27, "Output"]
}, Open  ]],
Cell[766143, 26101, 190, 5, 42, "Text"],
Cell[766336, 26108, 329, 6, 59, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

