(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.1'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    724198,      24082]*)
(*NotebookOutlinePosition[    726334,      24152]*)
(*  CellTagsIndexPosition[    725979,      24138]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["EventLocator", "Chapter"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"s:1"],

Cell["\<\
It is often useful to be able to detect and precisely locate  a \
change in a differential system. For example, with the detection of a \
singularity or state change, the appropriate action can be taken, such as \
restarting the integration. \
\>", "Text"],

Cell[TextData[{
  "An \"event\" for a differential system ",
  Cell[BoxData[
      \(TraditionalForm\`X' \((t)\)\  = \ f(t, \ X(t))\)]],
  " is a point along the solution at which a real valued event function ",
  Cell[BoxData[
      \(TraditionalForm\`g(t, \ X(t))\)]],
  " is zero.  It is also possible to consider boolean valued event functions, \
in which case the event occurs when the function changes from ",
  StyleBox["True", "MR"],
  " to ",
  StyleBox["False", "MR"],
  " or vice-versa."
}], "Text"],

Cell[TextData[{
  "The ",
  StyleBox["EventLocator", "MR"],
  " method which is built into ",
  StyleBox["NDSolve", "MR"],
  " works effectively as a controller method;  it handles checking for events \
and taking the appropriate action, but the integration of the differential \
system is otherwise left completely to an underlying method.  "
}], "Text"],

Cell[TextData[{
  "In this section, examples are given to demonstrate the basic use of the ",
  StyleBox["EventLocator", "MR"],
  " method and options.  Subsequent sections show more involved applications \
of event location, such as period detection,  Poincar\[EAcute] sections, and \
discontinuity handling."
}], "Text"],

Cell["\<\
These initialization commands load  some useful packages that have \
some differential equations to solve and define some utility functions.\
\>", \
"MathCaption"],

Cell[BoxData[{
    \(\(Needs["\<DifferentialEquations`NDSolveProblems`\>"];\)\), "\
\[IndentingNewLine]", 
    \(\(Needs["\<DifferentialEquations`NDSolveUtilities`\>"];\)\), "\
\[IndentingNewLine]", 
    \(\(Needs["\<DifferentialEquations`InterpolatingFunctionAnatomy`\>"];\)\),\
 "\[IndentingNewLine]", 
    \(\(Needs["\<Graphics`\>"];\)\), "\[IndentingNewLine]", 
    \(\(Needs["\<GUIKit`\>"];\)\)}], "Input",
  CellLabel->"In[6]:=",
  InitializationCell->True],

Cell["\<\
A simple example is locating an event, such as when a pendulum \
started at a non-equilibrium position will swing through its lowest point and \
stopping the integration at that point.\
\>", "Text"],

Cell["\<\
This integrates the pendulum equation up to the first point at \
which the solution x[t] crosses the axis.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol\  = \ 
      NDSolve[{\(x''\)[t]\  + \ Sin[x[t]]\  \[Equal] \ 
            0, \ \(x'\)[0]\  \[Equal] \ 0, \ x[0]\  \[Equal] \ 1}, 
        x, {t, 0, 10}, \ 
        Method \[Rule] {EventLocator, \ "\<Event\>" \[Rule] x[t]}]\)], "Input",\

  CellLabel->"In[5]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{"x", "\[Rule]", 
          
          TagBox[\(InterpolatingFunction[{{0.`, 1.6749939325948544`}}, 
              "<>"]\),
            False,
            Editable->False]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[5]="]
}, Open  ]],

Cell[TextData[{
  "From the solution you can see that pendulum reaches its lowest point x[t] \
= 0 at about t = 1.675.  Using the ",
  StyleBox["InterpolatingFunctionAnatomy", "MR"],
  " package, it is possible to extract the value from the",
  StyleBox[" InterpolatingFunction", "MR"],
  " object."
}], "Text"],

Cell["\<\
This extracts the point at which the event occurs and makes a plot \
of the solution (black) and its derivative (blue)  up to that point.\
\>", \
"MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(end\  = \ \(InterpolatingFunctionDomain[
            First[x\  /. \ 
                sol]]\)[\([1, \(-1\)]\)];\)\), "\[IndentingNewLine]", 
    \(Plot[Evaluate[{x[t], \ \(x'\)[t]}\  /. \ First[sol]], \ {t, 0, end}, \ 
      PlotStyle \[Rule] {{Black}, {Blue}}]\)}], "Input",
  CellLabel->"In[6]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.568588 0.302835 0.300484 [
[.16596 .29033 -12 -9 ]
[.16596 .29033 12 0 ]
[.3081 .29033 -9 -9 ]
[.3081 .29033 9 0 ]
[.45025 .29033 -12 -9 ]
[.45025 .29033 12 0 ]
[.5924 .29033 -3 -9 ]
[.5924 .29033 3 0 ]
[.73454 .29033 -12 -9 ]
[.73454 .29033 12 0 ]
[.87669 .29033 -9 -9 ]
[.87669 .29033 9 0 ]
[.01131 .00235 -12 -4.5 ]
[.01131 .00235 0 4.5 ]
[.01131 .15259 -24 -4.5 ]
[.01131 .15259 0 4.5 ]
[.01131 .45308 -18 -4.5 ]
[.01131 .45308 0 4.5 ]
[.01131 .60332 -6 -4.5 ]
[.01131 .60332 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.16596 .30283 m
.16596 .30908 L
s
[(0.25)] .16596 .29033 0 1 Mshowa
.3081 .30283 m
.3081 .30908 L
s
[(0.5)] .3081 .29033 0 1 Mshowa
.45025 .30283 m
.45025 .30908 L
s
[(0.75)] .45025 .29033 0 1 Mshowa
.5924 .30283 m
.5924 .30908 L
s
[(1)] .5924 .29033 0 1 Mshowa
.73454 .30283 m
.73454 .30908 L
s
[(1.25)] .73454 .29033 0 1 Mshowa
.87669 .30283 m
.87669 .30908 L
s
[(1.5)] .87669 .29033 0 1 Mshowa
.125 Mabswid
.05224 .30283 m
.05224 .30658 L
s
.08067 .30283 m
.08067 .30658 L
s
.1091 .30283 m
.1091 .30658 L
s
.13753 .30283 m
.13753 .30658 L
s
.19439 .30283 m
.19439 .30658 L
s
.22282 .30283 m
.22282 .30658 L
s
.25124 .30283 m
.25124 .30658 L
s
.27967 .30283 m
.27967 .30658 L
s
.33653 .30283 m
.33653 .30658 L
s
.36496 .30283 m
.36496 .30658 L
s
.39339 .30283 m
.39339 .30658 L
s
.42182 .30283 m
.42182 .30658 L
s
.47868 .30283 m
.47868 .30658 L
s
.50711 .30283 m
.50711 .30658 L
s
.53554 .30283 m
.53554 .30658 L
s
.56397 .30283 m
.56397 .30658 L
s
.62083 .30283 m
.62083 .30658 L
s
.64926 .30283 m
.64926 .30658 L
s
.67769 .30283 m
.67769 .30658 L
s
.70611 .30283 m
.70611 .30658 L
s
.76297 .30283 m
.76297 .30658 L
s
.7914 .30283 m
.7914 .30658 L
s
.81983 .30283 m
.81983 .30658 L
s
.84826 .30283 m
.84826 .30658 L
s
.90512 .30283 m
.90512 .30658 L
s
.93355 .30283 m
.93355 .30658 L
s
.96198 .30283 m
.96198 .30658 L
s
.99041 .30283 m
.99041 .30658 L
s
.25 Mabswid
0 .30283 m
1 .30283 L
s
.02381 .00235 m
.03006 .00235 L
s
[(-1)] .01131 .00235 1 0 Mshowa
.02381 .15259 m
.03006 .15259 L
s
[(-0.5)] .01131 .15259 1 0 Mshowa
.02381 .45308 m
.03006 .45308 L
s
[(0.5)] .01131 .45308 1 0 Mshowa
.02381 .60332 m
.03006 .60332 L
s
[(1)] .01131 .60332 1 0 Mshowa
.125 Mabswid
.02381 .0324 m
.02756 .0324 L
s
.02381 .06245 m
.02756 .06245 L
s
.02381 .0925 m
.02756 .0925 L
s
.02381 .12254 m
.02756 .12254 L
s
.02381 .18264 m
.02756 .18264 L
s
.02381 .21269 m
.02756 .21269 L
s
.02381 .24274 m
.02756 .24274 L
s
.02381 .27279 m
.02756 .27279 L
s
.02381 .33288 m
.02756 .33288 L
s
.02381 .36293 m
.02756 .36293 L
s
.02381 .39298 m
.02756 .39298 L
s
.02381 .42303 m
.02756 .42303 L
s
.02381 .48313 m
.02756 .48313 L
s
.02381 .51317 m
.02756 .51317 L
s
.02381 .54322 m
.02756 .54322 L
s
.02381 .57327 m
.02756 .57327 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .60332 m
.02499 .60332 L
.02605 .60332 L
.02729 .60331 L
.02846 .60331 L
.03053 .6033 L
.03279 .60329 L
.03527 .60327 L
.0379 .60324 L
.04262 .60318 L
.04749 .6031 L
.05205 .60301 L
.06244 .60274 L
.07305 .60237 L
.08274 .60196 L
.10458 .60077 L
.12357 .59943 L
.14429 .59765 L
.18493 .5932 L
.22406 .58773 L
.26565 .58064 L
.30571 .5726 L
.34426 .56376 L
.38527 .5532 L
.42475 .54194 L
.46273 .53013 L
.50315 .51655 L
.54206 .50255 L
.58342 .48671 L
.62326 .47059 L
.66159 .45434 L
.70238 .43634 L
.74164 .41837 L
.77939 .40059 L
.8196 .38119 L
.85828 .36215 L
.89942 .34162 L
.93905 .32164 L
.97619 .30283 L
s
0 0 1 r
.02381 .30283 m
.06244 .28566 L
.10458 .26698 L
.14415 .24954 L
.18221 .23289 L
.22272 .21538 L
.26171 .19876 L
.30316 .18142 L
.34309 .16509 L
.3815 .1498 L
.42237 .13404 L
.46172 .11944 L
.49955 .10601 L
.53984 .09243 L
.57861 .08015 L
.61984 .06801 L
.65954 .05731 L
.69774 .048 L
.73838 .03923 L
.77751 .03197 L
.81909 .02558 L
.85916 .02078 L
.89771 .01745 L
.91765 .01624 L
.92769 .01576 L
.93871 .01534 L
.94818 .01506 L
.9568 .01488 L
.96157 .01481 L
.96409 .01478 L
.96676 .01475 L
.96909 .01474 L
.97041 .01473 L
.97166 .01472 L
.9728 .01472 L
.97384 .01472 L
.97498 .01472 L
.97619 .01472 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[6]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 1}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`004`3oool500000?l0oooo2@3oool001D0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`3o0?ooo`<0oooo000E0?ooo`030000003oool0oooo00<0oooo0P00003o0?ooo`@0oooo
000=0?ooo`@00000103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0?l0oooo0`3o
ool001D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`3o0?ooo`<0oooo000D0?oo
o`8000001@3oool00`000000oooo0?ooo`3b0?ooo`T0003o1`3oool001/0oooo00<000000?ooo`3o
ool0h@3ooolA0000oa00oooo000K0?ooo`030000003oool0oooo0=H0oooo2`000?lQ0?ooo`006`3o
ool00`000000oooo0?ooo`3>0?ooo`P0003o;03oool001/0oooo00<000000?ooo`3oool0b03oool6
0000oc@0oooo000K0?ooo`800000a03oool50000ocX0oooo000K0?ooo`030000003oool0oooo0;h0
oooo1@000?lo0?ooo`006`3oool00`000000oooo0?ooo`2i0?ooo`D0003oA03oool001/0oooo00<0
00000?ooo`3oool0/`3oool60000odT0oooo000K0?ooo`030000003oool0oooo0:l0oooo10000?m?
0?ooo`006`3oool00`000000oooo0?ooo`2/0?ooo`<0003oD`3oool001/0oooo00<000000?ooo`3o
ool0Z03oool40000oeH0oooo000K0?ooo`030000003oool0oooo0:D0oooo0`000?mJ0?ooo`006`3o
ool200000:80oooo10000?mM0?ooo`006`3oool00`000000oooo0?ooo`2M0?ooo`@0003oH@3oool0
01/0oooo00<000000?ooo`3oool0VP3oool30000ofD0oooo000K0?ooo`030000003oool0oooo09H0
oooo10000?mX0?ooo`006`3oool00`000000oooo0?ooo`2B0?ooo`@0003oK03oool001/0oooo00<0
00000?ooo`3oool0S`3oool30000og00oooo000K0?ooo`030000003oool0oooo08`0oooo0`000?mc
0?ooo`006`3oool00`000000oooo0?ooo`280?ooo`@0003oMP3oool001/0oooo0P0000260?ooo`<0
003oNP3oool001/0oooo00<000000?ooo`3oool0PP3oool30000ogd0oooo000K0?ooo`030000003o
ool0oooo0800oooo0P000?n00?ooo`006`3oool00`000000oooo0?ooo`1m0?ooo`<0003oPP3oool0
01/0oooo00<000000?ooo`3oool0NP3oool30000ohD0oooo000K0?ooo`030000003oool0oooo07L0
oooo0`000?n80?ooo`006`3oool00`000000oooo0?ooo`1c0?ooo`@0003oR`3oool001/0oooo00<0
00000?ooo`3oool0L03oool30000ohl0oooo000K0?ooo`800000K`3oool20000oi80oooo000K0?oo
o`030000003oool0oooo06/0oooo0`000?nD0?ooo`006`3oool00`000000oooo0?ooo`1Y0?ooo`80
003oU`3oool001/0oooo00<000000?ooo`3oool0IP3oool30000oiT0oooo000K0?ooo`030000003o
ool0oooo06<0oooo0`000?nL0?ooo`006`3oool00`000000oooo0?ooo`1Q0?ooo`80003oW`3oool0
00P0oooo0P0000040?ooo`800000103oool2000000D0oooo00<000000?ooo`3oool0GP3oool30000
oj40oooo00070?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool00000103oool0
0`000000oooo0?ooo`1K0?ooo`<0003oY03oool000L0oooo00@000000?ooo`3oool000002`3oool0
0`000000oooo0?ooo`020?ooo`800000FP3oool20000ojL0oooo00000`3oool00000000000020000
0080oooo00@000000?ooo`3oool000002@3oool2000000D0oooo00<000000?ooo`3oool0EP3oool3
0000ojT0oooo00070?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`1D0?ooo`80003o[03oool000P0oooo0P00000:0?ooo`<00000103oool00`00
0000oooo0?ooo`1A0?ooo`<0003o[P3oool001/0oooo00<000000?ooo`3oool0C`3oool20000ok40
oooo000K0?ooo`030000003oool0oooo04d0oooo0P000?nc0?ooo`006`3oool00`000000oooo0?oo
o`1;0?ooo`80003o]@3oool001/0oooo00<000000?ooo`3oool0B@3oool20000okL0oooo000K0?oo
o`800000A`3oool30000okT0oooo000K0?ooo`030000003oool0oooo04<0oooo0`000?nl0?ooo`00
6`3oool00`000000oooo0?ooo`110?ooo`80003o_`3oool001/0oooo00<000000?ooo`3oool0?P3o
ool30000ol40oooo000K0?ooo`030000003oool0oooo03`0oooo0P000?o40?ooo`006`3oool00`00
0000oooo0?ooo`0j0?ooo`80003oaP3oool001/0oooo00<000000?ooo`3oool0>03oool20000olP0
oooo000K0?ooo`030000003oool0oooo03H0oooo0P000?o:0?ooo`006`3oool2000003D0oooo0P00
0?o<0?ooo`006`3oool00`000000oooo0?ooo`0a0?ooo`<0003ocP3oool001/0oooo00<000000?oo
o`3oool0;P3oool30000om40oooo000K0?ooo`030000003oool0oooo02`0oooo0P000?oD0?ooo`00
6`3oool00`000000oooo0?ooo`0Y0?ooo`<0003oeP3oool001/0oooo00<000000?ooo`3oool09`3o
ool20000omT0oooo000K0?ooo`030000003oool0oooo02D0oooo0P000?oK0?ooo`006`3oool00`00
0000oooo0?ooo`0S0?ooo`80003og@3oool001/0oooo0P00000R0?ooo`80003og`3oool001/0oooo
00<000000?ooo`3oool07`3oool20000on40oooo000K0?ooo`030000003oool0oooo01`0oooo0`00
0?oS0?ooo`006`3oool00`000000oooo0?ooo`0I0?ooo`<0003oiP3oool001/0oooo00<000000?oo
o`3oool05`3oool20000onT0oooo000K0?ooo`030000003oool0oooo01@0oooo0`000?l20?ooo`80
0000103oool2000000<0oooo100000030?ooo`800000503oool2000000@0oooo0P0000040?ooo`80
00005@3oool2000000@0oooo0P0000050?ooo`030000003oool0oooo0080oooo0P00000J0?ooo`D0
0000603oool500000080oooo0P0000030?ooo`@000000`3oool2000001@0oooo1@0000020?ooo`80
0000103oool2000001/0oooo000K0?ooo`030000003oool0oooo0140oooo0`000?l40?ooo`040000
003oool0oooo000000P0oooo00<000000?ooo`3oool00`3oool010000000oooo0?ooo`00000B0?oo
o`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000004`3oool010000000oooo0?oo
o`00000:0?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0I0?ooo`03
0000003oool0oooo01X0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`030?ooo`04
0000003oool0oooo000001D0oooo00<000000?ooo`3oool01`3oool010000000oooo0?ooo`00000J
0?ooo`006`3oool00`000000oooo0?ooo`0?0?ooo`80003o1`3oool010000000oooo0?ooo`000009
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0403oool010000000oooo0?ooo`00
000;0?ooo`030000003oool0oooo0140oooo00@000000?ooo`3oool000002P3oool00`000000oooo
0?ooo`040?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool06P3oool00`000000oooo
0?ooo`080?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04`3oool00`000000oooo
0?ooo`0:0?ooo`030000003oool0oooo01P0oooo000K0?ooo`8000003P3oool20000o`T0oooo00@0
00000?ooo`3oool000002P3oool00`000000oooo0?ooo`020?ooo`8000004`3oool010000000oooo
0?ooo`0000090?ooo`800000503oool010000000oooo0?ooo`00000;0?ooo`040000003oool0oooo
0?ooo`800000703oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool00P3oool2000001H0oooo00<000000?ooo`3oool0203oool2000001/0oooo000K0?oo
o`030000003oool0oooo00/0oooo0P000?l;0?ooo`040000003oool0oooo000000P0oooo00@00000
0?ooo`3oool000000`3oool00`000000oooo0?ooo`0B0?ooo`040000003oool0oooo000000T0oooo
00<000000?ooo`3oool04`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo0000
00<0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo
00L0oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0
oooo00P0oooo00<000000?ooo`3oool06P3oool001/0oooo00<000000?ooo`3oool02@3oool20000
o`h0oooo0P00000:0?ooo`800000103oool3000001<0oooo0P00000:0?ooo`<00000503oool20000
00T0oooo100000030?ooo`<000006P3oool2000001/0oooo0P00000:0?ooo`800000103oool30000
01@0oooo0P00000:0?ooo`<000006P3oool001/0oooo00<000000?ooo`3oool01`3oool20000ooT0
oooo000K0?ooo`030000003oool0oooo00D0oooo0P000?ok0?ooo`006`3oool00`000000oooo0?oo
o`030?ooo`80003oo@3oool001/0oooo00@000000?ooo`3oool0oooo0P000?oo0?ooo`006`3oool0
10000000oooo0000o`000?oo0?ooo`80oooo000D0?ooo`L000000P000?oo000000@00000000K0?oo
o`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?oo
o`040000003oool00000000000@0oooo00<000000?ooo`3oool00@3oool001/0oooo00<000000?oo
o`3oool08`3oool00`000000oooo0?ooo`0R0?ooo`030000003oool0oooo02<0oooo00<000000?oo
o`3oool08`3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo02<0oooo00<000000?oo
o`3oool04`3oool2000000X0oooo000K0?ooo`030000003oool0oooo0?@0oooo0P00000<0?ooo`00
6`3oool00`000000oooo0?ooo`3b0?ooo`8000003P3oool001/0oooo00<000000?ooo`3oool0l03o
ool200000100oooo000K0?ooo`030000003oool0oooo0>h0oooo0P00000B0?ooo`006`3oool00`00
0000oooo0?ooo`3/0?ooo`800000503oool001/0oooo0P00003[0?ooo`8000005P3oool001/0oooo
00<000000?ooo`3oool0j03oool2000001P0oooo000K0?ooo`030000003oool0oooo0>H0oooo0P00
000J0?ooo`006`3oool00`000000oooo0?ooo`3T0?ooo`800000703oool001/0oooo00<000000?oo
o`3oool0hP3oool2000001h0oooo000K0?ooo`030000003oool0oooo0>00oooo0P00000P0?ooo`00
6`3oool00`000000oooo0?ooo`3N0?ooo`8000008P3oool001/0oooo00<000000?ooo`3oool0g03o
ool2000002@0oooo000K0?ooo`800000g03oool00`000000oooo0?ooo`0T0?ooo`006`3oool00`00
0000oooo0?ooo`3I0?ooo`8000009`3oool001/0oooo00<000000?ooo`3oool0e`3oool2000002T0
oooo000K0?ooo`030000003oool0oooo0=D0oooo0P00000[0?ooo`006`3oool00`000000oooo0?oo
o`3C0?ooo`800000;@3oool001/0oooo00<000000?ooo`3oool0d03oool3000002l0oooo000K0?oo
o`030000003oool0oooo0<h0oooo0P00000b0?ooo`006`3oool00`000000oooo0?ooo`3<0?ooo`80
0000=03oool001/0oooo0P00003;0?ooo`800000=P3oool001/0oooo00<000000?ooo`3oool0b03o
ool2000003P0oooo000K0?ooo`030000003oool0oooo0<H0oooo0P00000j0?ooo`006`3oool00`00
0000oooo0?ooo`340?ooo`800000?03oool001/0oooo00<000000?ooo`3oool0`P3oool2000003h0
oooo000K0?ooo`030000003oool0oooo0<00oooo0P0000100?ooo`006`3oool00`000000oooo0?oo
o`2n0?ooo`800000@P3oool001/0oooo00<000000?ooo`3oool0^`3oool3000004@0oooo000K0?oo
o`800000^P3oool2000004L0oooo000K0?ooo`030000003oool0oooo0;L0oooo0P0000190?ooo`00
6`3oool00`000000oooo0?ooo`2e0?ooo`800000B`3oool001/0oooo00<000000?ooo`3oool0/`3o
ool2000004d0oooo000K0?ooo`030000003oool0oooo0;00oooo0`00001?0?ooo`006`3oool00`00
0000oooo0?ooo`2]0?ooo`<00000DP3oool000P0oooo0P0000040?ooo`800000103oool2000000D0
oooo00<000000?ooo`3oool0Z`3oool2000005D0oooo00070?ooo`040000003oool0oooo000000P0
oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`2X0?ooo`<00000E`3oool000L0
oooo00@000000?ooo`3oool000002`3oool00`000000oooo0?ooo`020?ooo`800000Y`3oool20000
05X0oooo00070?ooo`040000003oool0oooo000000T0oooo0P0000050?ooo`030000003oool0oooo
0:@0oooo0P00001L0?ooo`001`3oool010000000oooo0?ooo`0000090?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool0XP3oool2000005h0oooo00080?ooo`8000002P3oool3000000@0
oooo00<000000?ooo`3oool0X03oool200000600oooo000K0?ooo`030000003oool0oooo09h0oooo
0P00001R0?ooo`006`3oool00`000000oooo0?ooo`2K0?ooo`<00000I03oool001/0oooo00<00000
0?ooo`3oool0V@3oool2000006L0oooo000K0?ooo`030000003oool0oooo09H0oooo0`00001Y0?oo
o`006`3oool2000009D0oooo0P00001/0?ooo`006`3oool00`000000oooo0?ooo`2A0?ooo`<00000
KP3oool001/0oooo00<000000?ooo`3oool0SP3oool300000740oooo000K0?ooo`030000003oool0
oooo08`0oooo0P00001d0?ooo`006`3oool00`000000oooo0?ooo`290?ooo`<00000MP3oool001/0
oooo00<000000?ooo`3oool0QP3oool3000007T0oooo000K0?ooo`030000003oool0oooo08<0oooo
0`00001l0?ooo`006`3oool00`000000oooo0?ooo`210?ooo`800000O`3oool001/0oooo0P00001o
0?ooo`<00000P@3oool001/0oooo00<000000?ooo`3oool0N`3oool3000008@0oooo000K0?ooo`03
0000003oool0oooo07P0oooo0`0000270?ooo`006`3oool00`000000oooo0?ooo`1d0?ooo`@00000
RP3oool001/0oooo00<000000?ooo`3oool0L@3oool3000008h0oooo000K0?ooo`030000003oool0
oooo06h0oooo0`00002A0?ooo`006`3oool00`000000oooo0?ooo`1/0?ooo`800000U03oool001/0
oooo00<000000?ooo`3oool0J@3oool3000009H0oooo000K0?ooo`800000I`3oool3000009T0oooo
000K0?ooo`030000003oool0oooo06<0oooo0`00002L0?ooo`006`3oool00`000000oooo0?ooo`1O
0?ooo`@00000W`3oool001/0oooo00<000000?ooo`3oool0F`3oool400000:<0oooo000K0?ooo`03
0000003oool0oooo05D0oooo1P00002W0?ooo`006`3oool00`000000oooo0?ooo`1A0?ooo`@00000
[@3oool001/0oooo00<000000?ooo`3oool0CP3oool300000;40oooo000K0?ooo`030000003oool0
oooo04X0oooo1000002d0?ooo`006`3oool2000004L0oooo1000002h0?ooo`006`3oool00`000000
oooo0?ooo`100?ooo`H00000_03oool001/0oooo00<000000?ooo`3oool0>`3oool500000<80oooo
000K0?ooo`030000003oool0oooo03D0oooo1P0000370?ooo`006`3oool00`000000oooo0?ooo`0^
0?ooo`L00000c@3oool001/0oooo00<000000?ooo`3oool08`3oool;00000=@0oooo000C0?ooo`D0
00000`3oool00`000000oooo0?ooo`0K0?ooo`P00000g`3oool001D0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0@0?ooo`/00000i`3oool001D0oooo00<000000?ooo`3oool00`3o
oolC00000?80oooo000E0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0o`3oool3
0?ooo`005@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0?l0oooo0`3oool001@0
oooo0P0000050?ooo`030000003oool0oooo0?l0oooo0`3oool001/0oooo00<000000?ooo`3oool0
o`3oool30?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.180672, -1.10682, \
0.00661167, 0.0125109}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[7]="]
}, Open  ]],

Cell[TextData[{
  "When you use the event locator method, the events to be located and the \
action to take upon finding an event are specified through method options of \
the ",
  StyleBox["EventLocator", "MR"],
  " method."
}], "Text"],

Cell["\<\
The default action on detecting an event is to stop the integration as \
demonstrated above.  The event action can be any expression.  It is evaluated \
with numerical values substituted for the problem variables whenever an event \
is detected.\
\>", "Text"],

Cell[TextData[{
  "This prints the time and values each time the event  ",
  StyleBox["x'[t] = x[t]", "MR"],
  " is detected for a damped pendulum."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[{\(x''\)[t]\  + \  .1\ \(x'\)[t]\  + \ Sin[x[t]]\  \[Equal] \ 
          0, \ \(x'\)[0]\  \[Equal] \ 0, \ x[0]\  \[Equal] \ 1}, 
      x, {t, 0, 10}, \ 
      Method \[Rule] {EventLocator, \ "\<Event\>" \[Rule] \(x'\)[t] - \ 
              x[t], \ "\<EventAction\>" \[RuleDelayed] \ 
            Print["\<x'[\>", \ t, \ "\<] = x[\>", \ t, \ "\<] = \>", \ 
              x[t]]}]\)], "Input",
  CellLabel->"In[8]:="],

Cell[BoxData[
    InterpretationBox[\("x'["\[InvisibleSpace]2.498535243213698`\
\[InvisibleSpace]"] = \
x["\[InvisibleSpace]2.498535243213698`\[InvisibleSpace]"] = \
"\[InvisibleSpace]\(-0.5897530881423704`\)\),
      SequenceForm[ 
      "x'[", 2.498535243213698, "] = x[", 2.498535243213698, 
        "] = ", -.58975308814237037],
      Editable->False]], "Print",
  CellLabel->"From In[8]:="],

Cell[BoxData[
    InterpretationBox[\("x'["\[InvisibleSpace]5.787597844559808`\
\[InvisibleSpace]"] = \
x["\[InvisibleSpace]5.787597844559808`\[InvisibleSpace]"] = \
"\[InvisibleSpace]0.5012282621925576`\),
      SequenceForm[ 
      "x'[", 5.7875978445598077, "] = x[", 5.7875978445598077, 
        "] = ", .50122826219255756],
      Editable->False]], "Print",
  CellLabel->"From In[8]:="],

Cell[BoxData[
    InterpretationBox[\("x'["\[InvisibleSpace]9.034279806380399`\
\[InvisibleSpace]"] = \
x["\[InvisibleSpace]9.034279806380399`\[InvisibleSpace]"] = \
"\[InvisibleSpace]\(-0.42664521622527707`\)\),
      SequenceForm[ 
      "x'[", 9.0342798063803986, "] = x[", 9.0342798063803986, 
        "] = ", -.42664521622527707],
      Editable->False]], "Print",
  CellLabel->"From In[8]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{"x", "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 10.`}}, "<>"]\),
            False,
            Editable->False]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[8]="]
}, Open  ]],

Cell[TextData[{
  "Note that in the above example, the ",
  StyleBox["\"EventAction\"", "MR"],
  " option was given using ",
  StyleBox["RuleDelayed", "MR"],
  " (",
  "\[RuleDelayed]",
  ") to prevent it from evaluating except when the event is located."
}], "Text"],

Cell[TextData[{
  "You can see from the printed output that when the action does not stop the \
integration, multiple instances of an event can be detected.  Events are \
detected when the sign of the event expression changes.  You can restrict the \
event to be only for a sign change in a particular direction using the ",
  StyleBox["\"Direction\"", "MR"],
  "  option."
}], "Text"],

Cell[TextData[{
  "This collects the points at which the velocity changes from negative to \
positive for a damped driven pendulum.  ",
  StyleBox["Reap", "MR"],
  " and ",
  StyleBox["Sow", "MR"],
  "  are programming constructs which are useful for collecting data when you \
don't, at first, know how much data there will be.  ",
  StyleBox["Reap[", "MR"],
  StyleBox["expr", "TI"],
  StyleBox["]", "MR"],
  " gives the value of ",
  StyleBox["expr", "TI"],
  " together with all expressions to which ",
  StyleBox["Sow", "MR"],
  " has been applied during its evaluation.   Here ",
  StyleBox["Reap", "MR"],
  " ",
  StyleBox["encloses the use of ", "Text"],
  StyleBox["NDSolve", "MR"],
  StyleBox[" and ", "Text"],
  StyleBox["Sow", "MR"],
  " is a part of the event action, which allows us to collect data for each \
instance of an event."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Reap[
      NDSolve[{\(x''\)[t]\  + \  .1\ \(x'\)[t]\  + \ 
              Sin[x[t]]\  \[Equal] \  .1\ Cos[t], \ \(x'\)[0]\  \[Equal] \ 
            0, \ x[0]\  \[Equal] \ 1}, x, {t, 0, 50}, \ 
        Method \[Rule] {EventLocator, \ "\<Event\>" \[Rule] \(x'\)[
                t], \ "\<Direction\>" \[Rule] 
              1, \ "\<EventAction\>" \[RuleDelayed] \ 
              Sow[{t, \ x[t], \ \(x'\)[t]}]}]]\)], "Input",
  CellLabel->"In[9]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{"{", 
            RowBox[{"x", "\[Rule]", 
              TagBox[\(InterpolatingFunction[{{0.`, 50.`}}, "<>"]\),
                False,
                Editable->False]}], "}"}], "}"}], 
        ",", \({{{3.554068994663005`, \(-0.8793362332919002`\), 
              1.7295193055488767`*^-15}, {10.476207005448908`, \
\(-0.8322169054994999`\), 
              3.885780586188048`*^-16}, {17.185667623655466`, \
\(-0.8749387034665499`\), \(-3.3584246494910985`*^-15\)}, \
{23.772329986042138`, \(-0.9153517954793179`\), 
              1.790234627208065`*^-15}, {30.280460718449614`, \
\(-0.9271860535789482`\), 
              1.4753823163182744`*^-15}, {36.721717593150316`, \
\(-0.9108169586420907`\), \(-2.67841304690819`*^-15\)}, {43.101171581356034`, \
\(-0.8777079341664088`\), 
              2.7755575615628914`*^-17}, {49.42824299080917`, \
\(-0.8410832408324`\), \(-1.628211454551831`*^-14\)}}}\)}], "}"}]], "Output",
  CellLabel->"Out[9]="]
}, Open  ]],

Cell[TextData[{
  "You may notice from the output of the previous example that the events are \
detected when the derivative is only approximately zero.  When the method \
detects the presence of an event in a step of the underlying integrator (by a \
sign change of the event expression), then it uses a numerical method to \
approximately find the position of the root.  Since the location process is \
numerical, you should expect only approximate results.  Location method \
options ",
  StyleBox["AccuracyGoal", "MR"],
  ", ",
  StyleBox["PrecisionGoal", "MR"],
  ", and ",
  StyleBox["MaxIterations", "MR"],
  " can be given to those location methods which use ",
  StyleBox["FindRoot", "MR"],
  " to control tolerances for finding the root."
}], "Text"],

Cell["\<\
It is also desirable to be able to detect events which are the \
result of some modification which is external to the differential system \
itself, such as a user pressing a button.  It is often hard, if not \
impossible to represent such events as continuous functions with roots at the \
correct point.  Often, it is much more natural to reflect such a state change \
by a function which is either True or False.\
\>", "Text"],

Cell[TextData[{
  "For boolean valued event functions, an \"event\" occurs when the function \
switches from ",
  StyleBox["True", "MR"],
  " to ",
  StyleBox["False", "MR"],
  " or vice-versa.  The ",
  StyleBox["\"Direction\"", "MR"],
  " option can be used to restrict the event only from changes from ",
  StyleBox["True", "MR"],
  " to ",
  StyleBox["False", "MR"],
  " (",
  StyleBox["\"Direction\"->-1", "MR"],
  ") or only from changes from ",
  StyleBox["False", "MR"],
  " to ",
  StyleBox["True", "MR"],
  " (",
  StyleBox["\"Direction\"->1", "MR"],
  ")."
}], "Text"],

Cell["\<\
This opens up a small window with a button, which when pressed changes the \
value of the variable stop to True from its initialized value of False.\
\>", "MathCaption"],

Cell[BoxData[{
    \(\(NDSolve`stop\  = \ False;\)\), "\[IndentingNewLine]", 
    \(\(GUIRun[
        Widget["\<Panel\>", \ {Widget["\<Button\>", \
{\[IndentingNewLine]"\<label\>" \[Rule] "\<Stop\>", \
\[IndentingNewLine]BindEvent["\<action\>", \[IndentingNewLine]Script[
                    NDSolve`stop\  = \ True]]}]}]];\)\)}], "Input",
  CellLabel->"In[8]:="],

Cell["\<\
This integrates the pendulum equation up until the button is pushed (or the \
system runs out of memory)\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[{\(x''\)[t]\  + \ Sin[x[t]]\  \[Equal] \ 0, \ 
        x[0]\  \[Equal] \ 1, \ \(x'\)[0]\  \[Equal] \ 0}, \ 
      x, \ {t, \ 0, \ \[Infinity]}, 
      Method \[Rule] {EventLocator, \ "\<Event\>" \[RuleDelayed] \ 
            NDSolve`stop}, \ MaxSteps \[Rule] \[Infinity]]\)], "Input",
  CellLabel->"In[6]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{"x", "\[Rule]", 
          
          TagBox[\(InterpolatingFunction[{{0.`, 20316.257530863782`}}, 
              "<>"]\),
            False,
            Editable->False]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[6]="]
}, Open  ]],

Cell[TextData[{
  "Take note that in this example, the ",
  StyleBox["\"Event\"", "MR"],
  " option was specified with ",
  StyleBox["RuleDelayed", "MR"],
  " (",
  StyleBox[":>", "MR"],
  ") to prevent the immediate value of ",
  StyleBox["stop", "MR"],
  " from being evaluated and set up as the function. "
}], "Text"],

Cell["\<\
You can specify more than one event.  If the event function \
evaluates numerically to a list, then each component of the list is \
considered to be a separate event.  You can specify different actions, \
directions, etc. for each of these events by specifying the values of these \
options as lists of the appropriate length.  \
\>", "Text"],

Cell["\<\
This integrates the pendulum equation up until the point at which \
the above \"Stop\" button is pressed.  The number of complete swings of the \
pendulum is kept track of during the integration.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(stop\  = \ False;\)\), "\[IndentingNewLine]", 
    \(swings\  = \ 
      0; {\[IndentingNewLine]NDSolve[{\(x''\)[t]\  + Sin[x[t]]\  \[Equal] 
            0, \ x[0]\  \[Equal] \ 0, \ \(x'\)[0]\  \[Equal] \ 1}, \ 
        x, \ {t, \ 0, \ 1000000}, 
        Method \[Rule] {EventLocator, \ "\<Event\>" \[RuleDelayed] \ {x[t], 
                NDSolve`stop}, \ "\<EventAction\>" \[RuleDelayed] \ {\(swings\
++\), Throw[Null, \ "\<StopIntegration\>"]}, "\<Direction\>" \[Rule] {1, 
                All}}, \ MaxSteps \[Rule] Infinity], swings}\)}], "Input",
  CellLabel->"In[7]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{"{", 
            RowBox[{"x", "\[Rule]", 
              
              TagBox[\(InterpolatingFunction[{{0.`, 19115.344724226758`}}, 
                  "<>"]\),
                False,
                Editable->False]}], "}"}], "}"}], ",", "2834"}], 
      "}"}]], "Output",
  CellLabel->"Out[8]="]
}, Open  ]],

Cell["\<\
As you can see from the previous example, it is possible to mix \
real and boolean valued event functions.  The expected number of components \
and type of each component is based on the values at the initial condition \
and needs to be consistent throughout the integration.\
\>", "Text"],

Cell[TextData[{
  "The ",
  StyleBox["Method", "MR"],
  " option of ",
  StyleBox["EventLocator", "MR"],
  " allows the specification of the numerical method to use in the \
integration."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Event Location Methods", "Section",
  CellTags->"s:2"],

Cell[TextData[{
  "The ",
  StyleBox["EventLocator", "MR"],
  " method works by taking a step of the underlying method and checking to \
see if the sign (or parity) of any of the event functions is different at the \
step endpoints.  Event functions are expected to be real or boolean valued, \
so if there is a change, there must be an event in the step interval.  For \
each event function which has an event occurrence in a step, a refinement \
procedure is carried out to locate the position of the event within the \
interval."
}], "Text"],

Cell["\<\
There are several different methods which can be used refine the \
position. These include simply taking the solution at the beginning or the \
end of the integration interval, a linear interpolation of the event value \
and using bracketed root finding methods. The appropriate method to use \
depends on a trade off between execution speed and location accuracy.\
\>", \
"Text"],

Cell[TextData[{
  "If the event action is to stop the integration then the particular value \
at which the integration is stopped depends on the value obtained from the ",
  StyleBox["EventLocationMethod", "MR"],
  " option of ",
  StyleBox["EventLocator", "MR"],
  "."
}], "Text"],

Cell["\<\
Location of a single event is usually fast enough so that the \
method used will not significantly influence the overall computation time.  \
However, when an event is detected multiple times, the location refinement \
method can have a substantial effect.\
\>", "Text"],

Cell[CellGroupData[{

Cell["StepBegin and StepEnd Methods", "Subsection"],

Cell[TextData[{
  "The crudest methods are appropriate for when the exact position of the \
event location doesn't really matter or does not reflect anything with \
precision in the underlying calculation.  The stop button example from the \
previous section is such a case: time steps are computed so quickly that \
there is no way that one can time the press of a button to be within a \
particular time step, much less at a particular point within a time step.  \
Thus, based on the inherent accuracy of the event, there is no point in \
refining at all.  You can specify this by using the ",
  StyleBox["\"StepBegin", "MR"],
  "\" or ",
  StyleBox["\"StepEnd", "MR"],
  "\" location methods.  In any example where the definition of the event is \
heuristic or somewhat imprecise, this can be an appropriate choice."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["LinearInterpolation Method", "Subsection"],

Cell[TextData[{
  "When event results are needed for the purpose of points to plot in a \
graph, you only need to locate the event to the resolution of the graph.  \
While just using the step end is usually too crude for this, a single linear \
interpolation based on the event function values suffices.  You can specify \
refinement based on a single linear interpolation with the setting ",
  StyleBox["\"LinearInterpolation\"", "MR"],
  "."
}], "Text"],

Cell["\<\
This computes the solution for a single period of the pendulum \
equation and plots the solution for that period.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(sol\  = \ 
        First[NDSolve[{\(x''\)[t]\  + \ Sin[x[t]]\  \[Equal] \ 0, \ 
              x[0]\  \[Equal] \ 3, \ \(x'\)[0]\  \[Equal] \ 0}, 
            x, {t, 0, \[Infinity]}, \ 
            Method \[Rule] {"\<EventLocator\>", \[IndentingNewLine]\ "\<Event\
\>" \[Rule] \(x'\)[
                    t], \ \[IndentingNewLine]"\<EventAction\>" \[RuleDelayed] \
\ Throw[end\  = \ 
                      t, \ "\<StopIntegration\>"], "\<Direction\>" \[Rule] \
\(-1\), \ \[IndentingNewLine]"\<EventLocationMethod\>" -> \
"\<LinearInterpolation\>", 
                Method -> "\<ExplicitRungeKutta\>"}]];\)\), "\
\[IndentingNewLine]", 
    \(Plot[Evaluate[{x[t], \ \(x'\)[t]}\  /. \ sol], \ {t, 0, end}, \ 
      PlotStyle \[Rule] {{Black}, \ {Blue}}]\), "\[IndentingNewLine]", 
    \(\)}], "Input",
  CellLabel->"In[9]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0589559 0.309017 0.0981007 [
[.1712 .29652 -9 -9 ]
[.1712 .29652 9 0 ]
[.31859 .29652 -3 -9 ]
[.31859 .29652 3 0 ]
[.46598 .29652 -9 -9 ]
[.46598 .29652 9 0 ]
[.61337 .29652 -6 -9 ]
[.61337 .29652 6 0 ]
[.76076 .29652 -12 -9 ]
[.76076 .29652 12 0 ]
[.90815 .29652 -6 -9 ]
[.90815 .29652 6 0 ]
[.01131 .01471 -12 -4.5 ]
[.01131 .01471 0 4.5 ]
[.01131 .11282 -12 -4.5 ]
[.01131 .11282 0 4.5 ]
[.01131 .21092 -12 -4.5 ]
[.01131 .21092 0 4.5 ]
[.01131 .40712 -6 -4.5 ]
[.01131 .40712 0 4.5 ]
[.01131 .50522 -6 -4.5 ]
[.01131 .50522 0 4.5 ]
[.01131 .60332 -6 -4.5 ]
[.01131 .60332 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.1712 .30902 m
.1712 .31527 L
s
[(2.5)] .1712 .29652 0 1 Mshowa
.31859 .30902 m
.31859 .31527 L
s
[(5)] .31859 .29652 0 1 Mshowa
.46598 .30902 m
.46598 .31527 L
s
[(7.5)] .46598 .29652 0 1 Mshowa
.61337 .30902 m
.61337 .31527 L
s
[(10)] .61337 .29652 0 1 Mshowa
.76076 .30902 m
.76076 .31527 L
s
[(12.5)] .76076 .29652 0 1 Mshowa
.90815 .30902 m
.90815 .31527 L
s
[(15)] .90815 .29652 0 1 Mshowa
.125 Mabswid
.05329 .30902 m
.05329 .31277 L
s
.08277 .30902 m
.08277 .31277 L
s
.11224 .30902 m
.11224 .31277 L
s
.14172 .30902 m
.14172 .31277 L
s
.20068 .30902 m
.20068 .31277 L
s
.23016 .30902 m
.23016 .31277 L
s
.25963 .30902 m
.25963 .31277 L
s
.28911 .30902 m
.28911 .31277 L
s
.34807 .30902 m
.34807 .31277 L
s
.37755 .30902 m
.37755 .31277 L
s
.40702 .30902 m
.40702 .31277 L
s
.4365 .30902 m
.4365 .31277 L
s
.49546 .30902 m
.49546 .31277 L
s
.52493 .30902 m
.52493 .31277 L
s
.55441 .30902 m
.55441 .31277 L
s
.58389 .30902 m
.58389 .31277 L
s
.64285 .30902 m
.64285 .31277 L
s
.67232 .30902 m
.67232 .31277 L
s
.7018 .30902 m
.7018 .31277 L
s
.73128 .30902 m
.73128 .31277 L
s
.79024 .30902 m
.79024 .31277 L
s
.81971 .30902 m
.81971 .31277 L
s
.84919 .30902 m
.84919 .31277 L
s
.87867 .30902 m
.87867 .31277 L
s
.93763 .30902 m
.93763 .31277 L
s
.9671 .30902 m
.9671 .31277 L
s
.99658 .30902 m
.99658 .31277 L
s
.25 Mabswid
0 .30902 m
1 .30902 L
s
.02381 .01471 m
.03006 .01471 L
s
[(-3)] .01131 .01471 1 0 Mshowa
.02381 .11282 m
.03006 .11282 L
s
[(-2)] .01131 .11282 1 0 Mshowa
.02381 .21092 m
.03006 .21092 L
s
[(-1)] .01131 .21092 1 0 Mshowa
.02381 .40712 m
.03006 .40712 L
s
[(1)] .01131 .40712 1 0 Mshowa
.02381 .50522 m
.03006 .50522 L
s
[(2)] .01131 .50522 1 0 Mshowa
.02381 .60332 m
.03006 .60332 L
s
[(3)] .01131 .60332 1 0 Mshowa
.125 Mabswid
.02381 .03434 m
.02756 .03434 L
s
.02381 .05396 m
.02756 .05396 L
s
.02381 .07358 m
.02756 .07358 L
s
.02381 .0932 m
.02756 .0932 L
s
.02381 .13244 m
.02756 .13244 L
s
.02381 .15206 m
.02756 .15206 L
s
.02381 .17168 m
.02756 .17168 L
s
.02381 .1913 m
.02756 .1913 L
s
.02381 .23054 m
.02756 .23054 L
s
.02381 .25016 m
.02756 .25016 L
s
.02381 .26978 m
.02756 .26978 L
s
.02381 .2894 m
.02756 .2894 L
s
.02381 .32864 m
.02756 .32864 L
s
.02381 .34826 m
.02756 .34826 L
s
.02381 .36788 m
.02756 .36788 L
s
.02381 .3875 m
.02756 .3875 L
s
.02381 .42674 m
.02756 .42674 L
s
.02381 .44636 m
.02756 .44636 L
s
.02381 .46598 m
.02756 .46598 L
s
.02381 .4856 m
.02756 .4856 L
s
.02381 .52484 m
.02756 .52484 L
s
.02381 .54446 m
.02756 .54446 L
s
.02381 .56408 m
.02756 .56408 L
s
.02381 .5837 m
.02756 .5837 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
0 0 1 r
.5 Mabswid
.02381 .30902 m
.02498 .30874 L
.02615 .30847 L
.02848 .30792 L
.03316 .30681 L
.0425 .30455 L
.06277 .2992 L
.08169 .29317 L
.10025 .2857 L
.12037 .27505 L
.13915 .2618 L
.15951 .24269 L
.17949 .21816 L
.19813 .19016 L
.21834 .15686 L
.22777 .14212 L
.23249 .13541 L
.23721 .12933 L
.24183 .12415 L
.24415 .12188 L
.24646 .11986 L
.24877 .11808 L
.25108 .11657 L
.2534 .11534 L
.25455 .11483 L
.25571 .11439 L
.25696 .114 L
.25822 .11369 L
.25947 .11348 L
.26073 .11335 L
.26198 .11331 L
.26323 .11336 L
.26449 .11349 L
.26574 .11372 L
.267 .11403 L
.26825 .11443 L
.26951 .11491 L
.27076 .11548 L
.27327 .11688 L
.27578 .11859 L
.27812 .12046 L
.28046 .1226 L
.28514 .12756 L
.29451 .13982 L
.31481 .17222 L
.33473 .20424 L
.35332 .2299 L
.37348 .25195 L
.39229 .2676 L
.41268 .28031 L
Mistroke
.4327 .28959 L
.45137 .29626 L
.47162 .30208 L
.49052 .30677 L
.50905 .31114 L
.52915 .31613 L
.54791 .32152 L
.56825 .32879 L
.58821 .33816 L
.60683 .34974 L
.62702 .36648 L
.64587 .38703 L
.66435 .41228 L
.6844 .44435 L
.7031 .47457 L
.71324 .48846 L
.71831 .49414 L
.72338 .49874 L
.72575 .50047 L
.72693 .50123 L
.72812 .50192 L
.73048 .50308 L
.73167 .50355 L
.73285 .50394 L
.73403 .50425 L
.73522 .50448 L
.7364 .50464 L
.73758 .50472 L
.73877 .50472 L
.73995 .50464 L
.74113 .50448 L
.74232 .50424 L
.74348 .50393 L
.74464 .50355 L
.7458 .50309 L
.74696 .50257 L
.74928 .50129 L
.7516 .49975 L
.75624 .49587 L
.76088 .49103 L
.77095 .4779 L
.78102 .46235 L
.79981 .43133 L
.82018 .40045 L
.84018 .37578 L
.85883 .35796 L
.87905 .34339 L
.89794 .33321 L
.91645 .32557 L
.93653 .31906 L
Mistroke
.95527 .31405 L
.97558 .30918 L
.97619 .30904 L
Mfstroke
0 g
.02381 .60332 m
.02498 .60332 L
.02615 .60331 L
.02731 .60329 L
.02848 .60328 L
.02965 .60325 L
.03082 .60322 L
.03316 .60314 L
.03432 .6031 L
.03549 .60305 L
.03783 .60293 L
.0425 .60262 L
.04504 .60241 L
.04757 .60218 L
.05264 .60163 L
.0577 .60097 L
.06277 .60019 L
.07223 .59838 L
.08169 .5961 L
.09097 .59334 L
.10025 .58999 L
.11031 .58559 L
.12037 .5803 L
.13915 .56746 L
.15951 .54802 L
.17949 .52155 L
.19813 .48852 L
.21834 .44207 L
.23721 .38876 L
.25571 .32962 L
.27578 .26345 L
.29451 .20599 L
.31481 .15318 L
.33473 .11244 L
.35332 .08358 L
.37348 .06047 L
.39229 .04487 L
.40249 .03832 L
.41268 .03286 L
.42269 .02841 L
.4327 .02476 L
.44203 .02197 L
.4467 .02077 L
.45137 .0197 L
.46149 .01778 L
.46655 .01701 L
.47162 .01636 L
.47634 .01585 L
.4787 .01563 L
.48107 .01544 L
Mistroke
.48579 .01512 L
.48816 .015 L
.49052 .0149 L
.49168 .01485 L
.49283 .01482 L
.49399 .01479 L
.49515 .01476 L
.49631 .01474 L
.49747 .01473 L
.49863 .01472 L
.49978 .01472 L
.50094 .01472 L
.5021 .01472 L
.50326 .01474 L
.50442 .01475 L
.50557 .01478 L
.50673 .0148 L
.50789 .01484 L
.50905 .01488 L
.51031 .01493 L
.51156 .01498 L
.51408 .01511 L
.51659 .01526 L
.5191 .01544 L
.52413 .01589 L
.52915 .01644 L
.53384 .01705 L
.53853 .01777 L
.54322 .0186 L
.54791 .01953 L
.55808 .02198 L
.56825 .02505 L
.57823 .02875 L
.58821 .03325 L
.59752 .03827 L
.60683 .0442 L
.62702 .06087 L
.64587 .08239 L
.66435 .11067 L
.6844 .15114 L
.7031 .19896 L
.72338 .26047 L
.74232 .32281 L
.76088 .38265 L
.78102 .4402 L
.79981 .48414 L
.82018 .52091 L
.84018 .54757 L
.85883 .56575 L
.86894 .57346 L
Mistroke
.87905 .57992 L
.8885 .58497 L
.89794 .58922 L
.90719 .59269 L
.91645 .59557 L
.92649 .59809 L
.93151 .59914 L
.93653 .60005 L
.9459 .60144 L
.95059 .60198 L
.95527 .60243 L
.96035 .60281 L
.96289 .60296 L
.96543 .60308 L
.96797 .60318 L
.96924 .60322 L
.97051 .60325 L
.97177 .60328 L
.97304 .6033 L
.97431 .60331 L
.97558 .60332 L
.97619 .60332 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[9]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool000ooo`03003ooooo0?ooo`ooo`003ooo00<00?ooool0oooo3ooo0008ool20005ool00`00oooo
o`3oool?ool000Ooo`04003ooooo0004ool00`00ooooo`3oool?ool000[oo`03003ooooo00;oo`<0
07_oo`h008Goo`0000?oo`0000000P000ooo0P001Ooo00<00?ooool0M_oo1@003_oo1@00P?oo000:
ool00`00ooooo`02ool00`00ooooo`1`ool6000Hool5001kool000Ooo`04003ooooo0004ool00`00
ooooo`1^ool2000Sool3001hool000Soo`8000Goo`03003ooooo06_oo`<002Soo`<007Goo`003ooo
00<00?ooool0JOoo0P00;_oo0P00Looo000?ool2001Xool2000bool2001aool000ooo`03003ooooo
06Koo`03003ooooo03Coo`03003ooooo06koo`003ooo00<00?ooool0I?oo0P00>?oo0P00K_oo000?
ool00`00ooooo`1Sool00`00ooooo`0jool00`00ooooo`1[ool000ooo`03003ooooo06;oo`03003o
oooo03coo`03003ooooo06[oo`003ooo0P00HOoo0P00@?oo0P00J_oo000?ool00`00ooooo`1Oool0
0`00ooooo`12ool00`00ooooo`1Wool000ooo`03003ooooo05koo`03003ooooo04Coo`03003ooooo
06Koo`003ooo00<00?ooool0GOoo00<00?ooool0A_oo00<00?ooool0IOoo000?ool00`00ooooo`1L
ool00`00ooooo`18ool00`00ooooo`1Tool000ooo`03003ooooo05coo`03003ooooo04Soo`03003o
oooo06Coo`003ooo0P00G?oo00<00?ooool0B_oo00<00?ooool0Hooo000?ool00`00ooooo`1Jool0
0`00ooooo`1<ool00`00ooooo`1Rool000ooo`03003ooooo05Woo`03003ooooo04koo`03003ooooo
067oo`003ooo00<00?ooool0F?oo00<00?ooool0D?oo00<00?ooool0H?oo000?ool00`00ooooo`1H
ool00`00ooooo`1@ool00`00ooooo`1Pool000ooo`8005Soo`03003ooooo05;oo`03003ooooo05oo
o`003ooo00<00?ooool0E_oo00<00?ooool0Dooo00<00?ooool0Gooo000?ool00`00ooooo`1Eool0
0`00ooooo`1Eool00`00ooooo`1Nool000Ooo`@000Coo`03003ooooo05Goo`03003ooooo05Koo`03
003ooooo05goo`001ooo00<00?ooool01Ooo00<00?ooool0E?oo00<00?ooool0Eooo00<00?ooool0
GOoo0008ool00`00ooooo`04ool3001Cool00`00ooooo`1Iool00`00ooooo`1Lool00003ool00000
008000Coo`03003ooooo00?oo`03003ooooo03goo`H07a3oo`03003ooooo05[oo`03003ooooo05_o
o`002_oo00<00?ooool00_oo00<00?ooool0>ooo0P0O1Ooo0P0O3_oo00<00?ooool0Fooo00<00?oo
ool0Fooo0007ool01000ooooo`001?oo00<00?ooool0>_oo00<07oooool01ooo00<07oooool02ooo
00<00?ooool0G?oo00<00?ooool0F_oo0008ool20005ool00`00ooooo`0iool201l:ool00`0Ooooo
o`09ool00`00ooooo`1Mool00`00ooooo`1Jool000ooo`03003ooooo03Soo`0301oooooo00_oo`03
01oooooo00Soo`03003ooooo05koo`03003ooooo05Woo`003ooo0P00>Ooo00<07oooool03?oo00<0
7oooool01ooo00<00?ooool0G_oo00<00?ooool0FOoo000?ool00`00ooooo`0gool00`0Oooooo`0=
ool00`0Oooooo`06ool00`00ooooo`1Pool00`00ooooo`1Hool000ooo`03003ooooo03Ooo`0301oo
oooo00koo`0301oooooo00Goo`03003ooooo063oo`03003ooooo05Soo`003ooo00<00?ooool0=_oo
00<07oooool04?oo00<07oooool00ooo00<00?ooool0H_oo00<00?ooool0Eooo000?ool00`00oooo
o`0eool00`0Oooooo`0Aool00`0Oooooo`03ool00`00ooooo`1Rool00`00ooooo`1Gool000ooo`80
03Koo`0301oooooo01;oo`0501ooooooool0001Vool00`00ooooo`1Fool000ooo`03003ooooo03Co
o`0301oooooo01Coo`0401oooooo001Vool00`00ooooo`1Fool000ooo`03003ooooo03?oo`0301oo
oooo01Goo`0401oooooo001Wool00`00ooooo`1Eool000ooo`03003ooooo03;oo`0301oooooo01Oo
o`0301l00?oo06Ooo`03003ooooo05Goo`003ooo00<00?ooool0<_oo00<07oooool06?oo00<00?oo
ool0Iooo00<00?ooool0E?oo000?ool00`00ooooo`0aool00`0Oooooo`0Hool00`0001ooo`1Xool0
0`00ooooo`1Dool000ooo`80037oo`0301oooooo01Woo`03003oo`0O06Soo`03003ooooo05Coo`00
3ooo00<00?ooool0<?oo00<07oooool06?oo00D00?ooooooo`0O06Soo`03003ooooo05?oo`003ooo
00<00?ooool0;ooo00<07oooool06Ooo00D00?ooooooo`0O06Soo`03003ooooo05?oo`003ooo00<0
0?ooool0;_oo00<07oooool06_oo00<00?ooool00_oo00<07oooool0IOoo00<00?ooool0Dooo000?
ool00`00ooooo`0^ool00`0Oooooo`0Iool00`00ooooo`03ool00`0Oooooo`1Vool00`00ooooo`1B
ool000ooo`8002koo`0301oooooo01[oo`03003ooooo00Coo`0301oooooo06Goo`03003ooooo05;o
o`003ooo00<00?ooool0;?oo00<07oooool06_oo00<00?ooool01Ooo00<07oooool0I_oo00<00?oo
ool0DOoo000?ool00`00ooooo`0/ool00`0Oooooo`0Jool00`00ooooo`06ool00`0Oooooo`1Uool0
0`00ooooo`1Aool000ooo`03003ooooo02_oo`0301oooooo01[oo`03003ooooo00Ooo`0301oooooo
06Goo`03003ooooo057oo`001ooo10001?oo00<00?ooool0:ooo00<07oooool06_oo00<00?ooool0
2?oo00<07oooool0IOoo00<00?ooool0D?oo0009ool00`00ooooo`03ool00`00ooooo`0Zool00`0O
ooooo`0Kool00`00ooooo`09ool00`0Oooooo`1Tool00`00ooooo`1@ool000Woo`03003ooooo00?o
o`<002Woo`0301oooooo01_oo`03003ooooo00_oo`0301oooooo06?oo`03003ooooo053oo`0000?o
o`0000000P001?oo00<00?ooool00ooo00<00?ooool0:Ooo00<07oooool06ooo00<00?ooool03?oo
00<07oooool0Hooo00<00?ooool0Cooo0009ool00`00ooooo`03ool00`00ooooo`0Xool00`0Ooooo
o`0Lool00`00ooooo`0<ool00`0Oooooo`1Sool00`00ooooo`1?ool000Ooo`<000Goo`03003ooooo
02Ooo`0301oooooo01coo`03003ooooo00koo`0301oooooo06;oo`03003ooooo04ooo`002Ooo00<0
0?ooool00ooo00<00?ooool09ooo00<07oooool07?oo00<00?ooool03ooo00<07oooool0H_oo00<0
0?ooool0C_oo000?ool2000Wool00`0Oooooo`0Mool00`00ooooo`0@ool00`0Oooooo`1Qool00`00
ooooo`1>ool000ooo`03003ooooo02Goo`0301oooooo01koo`03003ooooo017oo`0301oooooo063o
o`03003ooooo04koo`003ooo00<00?ooool09?oo00<07oooool07_oo00<00?ooool04ooo00<07ooo
ool0H?oo00<00?ooool0COoo000?ool00`00ooooo`0Tool00`0Oooooo`0Nool00`00ooooo`0Cool0
0`0Oooooo`1Pool00`00ooooo`1=ool000ooo`03003ooooo02?oo`0301oooooo01ooo`03003ooooo
01Coo`0301oooooo05ooo`03003ooooo04goo`003ooo00<00?ooool08_oo00<07oooool07ooo00<0
0?ooool05_oo00<07oooool0Gooo00<00?ooool0C?oo000?ool2000Qool201lRool00`00ooooo`0G
ool00`0Oooooo`1Nool00`00ooooo`1<ool000ooo`03003ooooo01ooo`0301oooooo02;oo`03003o
oooo01Soo`0301oooooo05goo`03003ooooo04coo`003ooo00<00?ooool07_oo00<07oooool08_oo
00<00?ooool06_oo00<07oooool0GOoo00<00?ooool0Booo000?ool00`00ooooo`0Mool00`0Ooooo
o`0Sool00`00ooooo`0Kool00`0Oooooo`1Lool00`00ooooo`1;ool000ooo`03003ooooo01coo`03
01oooooo02Coo`03003ooooo01coo`0301oooooo05_oo`03003ooooo04_oo`003ooo0P006ooo0P0O
9_oo00<00?ooool07_oo00<07oooool0Fooo00<00?ooool0B_oo000?ool00`00ooooo`0Iool00`0O
ooooo`02ool40003ool00`00ooooo`02ool3000Eool00`00ooooo`08ool3000Dool201l;ool00`00
ooooo`03ool00`00ooooo`02ool3000Mool40003ool2000Jool40002ool40003ool00`00ooooo`02
ool3000Jool40002ool3000Gool000ooo`03003ooooo01Ooo`807`Goo`03003ooooo00coo`03003o
oooo01;oo`03003ooooo00_oo`03003ooooo01?oo`807`Woo`03003ooooo00_oo`03003ooooo01co
o`05003oooooool00002ool00`00ooooo`0Iool01@00oooooooo00003_oo00<00?ooool06Ooo00<0
0?ooool01?oo00<00?ooool05?oo000?ool00`00ooooo`0Eool201l8ool00`00ooooo`0;ool00`00
ooooo`0Aool00`00ooooo`0<ool00`00ooooo`0Eool201l8ool00`00ooooo`0:ool00`00ooooo`0L
ool01@00oooooooo00000_oo00<00?ooool06Ooo00<00?ooool00_oo00<00?ooool02ooo00<00?oo
ool06Ooo00<00?ooool01?oo00<00?ooool05?oo000?ool00`00ooooo`0Cool201l;ool00`00oooo
o`07ool3000Dool00`00ooooo`09ool3000Jool201l6ool00`00ooooo`07ool3000Oool01@00oooo
oooo00000_oo00<00?ooool06Ooo0P001?oo00<00?ooool01ooo0`007?oo00@00?ooooooo`<001Oo
o`003ooo00<00?ooool04Ooo0P0O3_oo00<00?ooool01_oo00<00?ooool05?oo00<00?ooool02Ooo
00<00?ooool07?oo0P0O1Ooo00<00?ooool01_oo00<00?ooool07ooo00D00?ooooooo`0000;oo`03
003ooooo01Woo`8000Goo`03003ooooo00Koo`03003ooooo01coo`05003oooooool0000Iool000oo
o`80013oo`807`goo`04003ooooo0008ool00`00ooooo`0Cool00`00ooooo`0:ool00`00ooooo`0N
ool401l00ooo003oo`07ool00`00ooooo`0Mool30003ool01000ooooo`006Ooo10000_oo00@00?oo
ool000Soo`03003ooooo01[oo`<000?oo`03003ooooo01Ooo`003ooo00<00?ooool02ooo100O4?oo
0P002Ooo10004_oo00<00?ooool02_oo10007ooo0P00100O1_oo10007_oo00<00?ooool00_oo0P00
7?oo00<00?oo00000_oo0P002Ooo10006ooo00@00?ooooooo`@001Koo`003ooo00<00?ooool01ooo
100O=Ooo00<00?ooool0<ooo0`0OB_oo00<00?ooool0B?oo000?ool00`00ooooo`04ool301liool0
0`00ooooo`0fool401m6ool00`00ooooo`18ool000ooo`03003oo`0O00@07c_oo`03003ooooo03_o
o`@07d?oo`03003ooooo04Ooo`002?oo1`000P0OPP000P0OP000100O1`00000?ool00`00ooooo`05
ool00`00ooooo`05ool00`00ooooo`06ool00`00ooooo`05ool00`00ooooo`05ool00`00ooooo`05
ool00`00ooooo`05ool00`00ooooo`06ool00`00ooooo`05ool00`00ooooo`05ool00`00ooooo`05
ool00`00ooooo`05ool00`00ooooo`06ool00`00ooooo`05ool00`00ooooo`05ool00`00ooooo`05
ool00`00ooooo`0401l00ooo003oo`07ool00`00ooooo`05ool00`00ooooo`05ool00`00ooooo`05
ool00`00ooooo`05ool00`00ooooo`06ool00`00ooooo`05ool00`00ool00005ool00`00ooooo`05
ool00`00ooooo`05ool00`00ooooo`06ool00`00ooooo`05ool00`00ooooo`05ool00`00ooooo`05
ool01000oooooooo100O00<00?ooool01_oo0@000Ooo000?ool00`00ooooo`0Vool00`00ooooo`0E
ool00`00ooooo`0>ool00`00ooooo`0Vool00`00ooooo`0<ool501lEool00`00ooooo`0Pool00`00
ooooo`03ool00`00ooooo`0Vool00`00ooooo`07ool201l?ool000ooo`03003ooooo03koo`03003o
oooo04_oo`807cOoo`03003ooooo03;oo`<07a7oo`003ooo00<00?ooool0?_oo00<00?ooool0COoo
0`0O=?oo00<00?ooool0;ooo0`0O5?oo000?ool2000nool00`00ooooo`1Aool201lbool00`00oooo
o`0/ool301lGool000ooo`03003ooooo03goo`03003ooooo05?oo`<07c3oo`03003ooooo02Woo`80
7a[oo`003ooo00<00?ooool0?Ooo00<00?ooool0E_oo0`0O;Ooo00<00?ooool09ooo0P0O7?oo000?
ool00`00ooooo`0lool00`00ooooo`1Jool201l[ool00`00ooooo`0Uool201lNool000ooo`03003o
oooo03coo`03003ooooo05coo`807bWoo`03003ooooo02?oo`807b3oo`003ooo00<00?ooool0??oo
00<00?ooool0G_oo0P0O:?oo00<00?ooool08Ooo00<07oooool08?oo000?ool2000lool00`00oooo
o`1Qool00`0Oooooo`0Uool00`00ooooo`0Oool201lSool000ooo`03003ooooo03_oo`03003ooooo
06;oo`0301oooooo02Coo`03003ooooo01koo`0301oooooo02?oo`003ooo00<00?ooool0>ooo00<0
0?ooool0Hooo00<07oooool09?oo00<00?ooool06ooo0P0O9_oo000?ool00`00ooooo`0kool00`00
ooooo`1Tool201lTool00`00ooooo`0Jool00`0Oooooo`0Vool000ooo`03003ooooo03[oo`03003o
oooo06Ooo`0301oooooo027oo`03003ooooo01Woo`0301oooooo02Ooo`003ooo0P00>ooo00<00?oo
ool0J?oo00<07oooool08Ooo00<00?ooool05ooo00<07oooool0:?oo000?ool00`00ooooo`0jool0
0`00ooooo`1Yool00`0Oooooo`0Pool00`00ooooo`0Fool00`0Oooooo`0Yool000ooo`03003ooooo
03Woo`03003ooooo06_oo`0301oooooo01ooo`03003ooooo01Goo`0301oooooo02[oo`003ooo00<0
0?ooool0>Ooo00<00?ooool0Jooo00<07oooool07ooo00<00?ooool05?oo00<07oooool0:ooo000?
ool00`00ooooo`0iool00`00ooooo`1/ool00`0Oooooo`0Oool00`00ooooo`0Bool00`0Oooooo`0/
ool000ooo`03003ooooo03Soo`03003ooooo06koo`0301oooooo01koo`03003ooooo017oo`0301oo
oooo02goo`003ooo0P00>Ooo00<00?ooool0Kooo00<07oooool07Ooo00<00?ooool04?oo00<07ooo
ool0;_oo000?ool00`00ooooo`0hool00`00ooooo`1`ool00`0Oooooo`0Mool00`00ooooo`0>ool0
0`0Oooooo`0_ool000ooo`03003ooooo03Ooo`03003ooooo077oo`0301oooooo01goo`03003ooooo
00goo`0301oooooo033oo`001ooo10001?oo00<00?ooool0=ooo00<00?ooool0L_oo00<07oooool0
7?oo00<00?ooool03?oo00<07oooool0<Ooo0009ool00`00ooooo`03ool00`00ooooo`0gool00`00
ooooo`1cool00`0Oooooo`0Lool00`00ooooo`0:ool00`0Oooooo`0bool000Woo`03003ooooo00?o
o`<003Koo`03003ooooo07Goo`0301oooooo01_oo`03003ooooo00[oo`0301oooooo03;oo`002Ooo
00<00?ooool00ooo00<00?ooool0=_oo00<00?ooool0MOoo00<07oooool07?oo00<00?ooool02?oo
00<07oooool0<ooo0009ool00`00ooooo`03ool00`00ooooo`0fool00`00ooooo`1fool00`0Ooooo
o`0Kool00`00ooooo`08ool00`0Oooooo`0cool000Ooo`<000Goo`03003ooooo03Goo`03003ooooo
07Soo`0301oooooo01[oo`03003ooooo00Ooo`0301oooooo03Coo`002Ooo00<00?ooool00ooo00<0
0?ooool0=Ooo00<00?ooool0N?oo00<07oooool06ooo00<00?ooool01_oo00<07oooool0=?oo000?
ool00`00ooooo`0eool00`00ooooo`1iool00`0Oooooo`0Jool00`00ooooo`05ool00`0Oooooo`0e
ool000ooo`8003Goo`03003ooooo07_oo`0301oooooo01Woo`03003ooooo00Goo`0301oooooo03Go
o`003ooo00<00?ooool0=?oo00<00?ooool0Nooo00<07oooool06_oo00<00?ooool00ooo00<07ooo
ool0=_oo000?ool00`00ooooo`0dool00`00ooooo`1lool00`0Oooooo`0Iool00`00ooooo`02ool0
0`0Oooooo`0gool000ooo`03003ooooo03?oo`03003ooooo07koo`0301oooooo01Woo`05003ooooo
ool07`0iool000ooo`03003ooooo03?oo`03003ooooo07koo`0301oooooo01Woo`04003ooooo01lj
ool000ooo`8003Coo`03003ooooo07ooo`0301oooooo01Soo`04003ooooo01ljool000ooo`03003o
oooo03;oo`03003ooooo087oo`0301oooooo01Soo`0300007ooo03[oo`003ooo00<00?ooool0<_oo
00<00?ooool0POoo00<07oooool06?oo00<0000Oool0>_oo000?ool00`00ooooo`0aool00`00oooo
o`23ool00`0Oooooo`0Gool00`0O003oo`0jool000ooo`03003ooooo037oo`03003ooooo08?oo`03
01oooooo01Ooo`0301l00?oo03[oo`003ooo00<00?ooool0<?oo00<00?ooool0QOoo00<07oooool0
5Ooo00<07ooo0000>ooo000?ool2000aool00`00ooooo`26ool00`0Oooooo`0Cool01@0Ooooooooo
0000>_oo000?ool00`00ooooo`0_ool00`00ooooo`27ool00`0Oooooo`0Cool01@0Ooooooooo0000
>_oo000?ool00`00ooooo`0_ool00`00ooooo`28ool00`0Oooooo`0Aool00`0Oooooo`03ool00`00
ooooo`0gool000ooo`03003ooooo02koo`03003ooooo08[oo`0301oooooo00ooo`0301oooooo00Co
o`03003ooooo03Ooo`003ooo00<00?ooool0;_oo00<00?ooool0R_oo00<07oooool03_oo00<07ooo
ool01_oo00<00?ooool0=_oo000?ool2000^ool00`00ooooo`2<ool00`0Oooooo`0=ool00`0Ooooo
o`06ool00`00ooooo`0fool000ooo`03003ooooo02goo`03003ooooo08goo`0301oooooo00_oo`03
01oooooo00Ooo`03003ooooo03Koo`003ooo00<00?ooool0;?oo00<00?ooool0S_oo00<07oooool0
2_oo00<07oooool02Ooo00<00?ooool0=Ooo000?ool00`00ooooo`0/ool00`00ooooo`2?ool00`0O
ooooo`07ool201l<ool00`00ooooo`0eool000Ooo`@000Coo`03003ooooo02_oo`03003ooooo097o
o`<07`Goo`0301oooooo00goo`03003ooooo03Coo`001ooo00<00?ooool01Ooo00<00?ooool0:ooo
00<00?ooool0Tooo1`0O3ooo00<00?ooool0=?oo0008ool00`00ooooo`04ool3000Zool00`00oooo
o`2[ool00`00ooooo`0cool000Woo`03003ooooo00?oo`03003ooooo02[oo`03003ooooo0:_oo`03
003ooooo03?oo`002_oo00<00?ooool00_oo00<00?ooool0:Ooo00<00?ooool0[Ooo00<00?ooool0
<_oo0007ool01000ooooo`001?oo00<00?ooool0:Ooo00<00?ooool0[Ooo00<00?ooool0<_oo0008
ool20005ool00`00ooooo`0Xool00`00ooooo`2_ool00`00ooooo`0aool000ooo`8002Soo`03003o
oooo0;7oo`03003ooooo033oo`003ooo00<00?ooool09ooo00<00?ooool0/_oo00<00?ooool0;ooo
000?ool00`00ooooo`0Vool00`00ooooo`2dool00`00ooooo`0^ool000ooo`03003ooooo02Koo`03
003ooooo0;Coo`03003ooooo02koo`003ooo00<00?ooool09Ooo00<00?ooool0]_oo00<00?ooool0
;Ooo000?ool00`00ooooo`0Tool00`00ooooo`2hool00`00ooooo`0/ool000ooo`8002Goo`03003o
oooo0;Woo`03003ooooo02_oo`003ooo00<00?ooool08ooo00<00?ooool0^ooo00<00?ooool0:_oo
000?ool00`00ooooo`0Rool00`00ooooo`2mool00`00ooooo`0Yool000ooo`03003ooooo023oo`80
0<7oo`03003ooooo02Soo`003ooo00<00?ooool07ooo00<00?ooool0`_oo00<00?ooool09ooo000?
ool2000Oool00`00ooooo`34ool00`00ooooo`0Vool000ooo`03003ooooo01goo`03003ooooo0<Ko
o`03003ooooo02Goo`003ooo00<00?ooool06ooo0P00b_oo00<00?ooool09?oo000?ool00`00oooo
o`0Iool2003=ool2000Tool000ooo`03003ooooo01Soo`03003ooooo0<ooo`8002;oo`003ooo0P00
5ooo0P00e?oo00<00?ooool07ooo000?ool00`00ooooo`0Dool2003Gool2000Oool000ooo`03003o
oooo017oo`<00=_oo`<001coo`003ooo00<00?ooool03ooo0P00hOoo0P006_oo0008ool20005ool0
0`00ooooo`0<ool3003Uool5000Eool000Ooo`04003ooooo0004ool00`00ooooo`05ool7003]ool5
000@ool000[oo`03003ooooo00;oo`P00?Woo`T000Ooo`002?oo0P001Ooo00<00?ooool0oooo3ooo
000:ool00`00ooooo`02ool00`00ooooo`3oool?ool000Ooo`04003ooooo0004ool00`00ooooo`3o
ool?ool000Soo`8000Goo`03003ooooo0?ooo`ooo`00oooo8Ooo003ooolQool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.943887, -3.24229, \
0.0609824, 0.0366488}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[10]="]
}, Open  ]],

Cell["\<\
At the resolution of the plot over the entire period, you cannot \
see that the endpoint may not be exactly where the derivative hits the axis.  \
However, if you zoom in enough, you can see the error.\
\>", "Text"],

Cell["This shows a plot just near the endpoint.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot[
      Evaluate[\(x'\)[t]\  /. \ sol], \ {t, end*\((1\  - \  .001)\), end}, \ 
      PlotStyle \[Rule] Blue]\)], "Input",
  CellLabel->"In[11]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-951.405 58.9559 0.0147151 237.325 [
[.29108 .00222 -21 -9 ]
[.29108 .00222 21 0 ]
[.43847 .00222 -18 -9 ]
[.43847 .00222 18 0 ]
[.58586 .00222 -21 -9 ]
[.58586 .00222 21 0 ]
[.73325 .00222 -15 -9 ]
[.73325 .00222 15 0 ]
[.88064 .00222 -21 -9 ]
[.88064 .00222 21 0 ]
[.13119 .13338 -36 -4.5 ]
[.13119 .13338 0 4.5 ]
[.13119 .25204 -30 -4.5 ]
[.13119 .25204 0 4.5 ]
[.13119 .3707 -36 -4.5 ]
[.13119 .3707 0 4.5 ]
[.13119 .48936 -30 -4.5 ]
[.13119 .48936 0 4.5 ]
[.13119 .60803 -36 -4.5 ]
[.13119 .60803 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.29108 .01472 m
.29108 .02097 L
s
[(16.1425)] .29108 .00222 0 1 Mshowa
.43847 .01472 m
.43847 .02097 L
s
[(16.145)] .43847 .00222 0 1 Mshowa
.58586 .01472 m
.58586 .02097 L
s
[(16.1475)] .58586 .00222 0 1 Mshowa
.73325 .01472 m
.73325 .02097 L
s
[(16.15)] .73325 .00222 0 1 Mshowa
.88064 .01472 m
.88064 .02097 L
s
[(16.1525)] .88064 .00222 0 1 Mshowa
.125 Mabswid
.02578 .01472 m
.02578 .01847 L
s
.05526 .01472 m
.05526 .01847 L
s
.08474 .01472 m
.08474 .01847 L
s
.11422 .01472 m
.11422 .01847 L
s
.17317 .01472 m
.17317 .01847 L
s
.20265 .01472 m
.20265 .01847 L
s
.23213 .01472 m
.23213 .01847 L
s
.26161 .01472 m
.26161 .01847 L
s
.32056 .01472 m
.32056 .01847 L
s
.35004 .01472 m
.35004 .01847 L
s
.37952 .01472 m
.37952 .01847 L
s
.40899 .01472 m
.40899 .01847 L
s
.46795 .01472 m
.46795 .01847 L
s
.49743 .01472 m
.49743 .01847 L
s
.52691 .01472 m
.52691 .01847 L
s
.55638 .01472 m
.55638 .01847 L
s
.61534 .01472 m
.61534 .01847 L
s
.64482 .01472 m
.64482 .01847 L
s
.6743 .01472 m
.6743 .01847 L
s
.70377 .01472 m
.70377 .01847 L
s
.76273 .01472 m
.76273 .01847 L
s
.79221 .01472 m
.79221 .01847 L
s
.82169 .01472 m
.82169 .01847 L
s
.85116 .01472 m
.85116 .01847 L
s
.91012 .01472 m
.91012 .01847 L
s
.9396 .01472 m
.9396 .01847 L
s
.96908 .01472 m
.96908 .01847 L
s
.99855 .01472 m
.99855 .01847 L
s
.25 Mabswid
0 .01472 m
1 .01472 L
s
.14369 .13338 m
.14994 .13338 L
s
[(0.0005)] .13119 .13338 1 0 Mshowa
.14369 .25204 m
.14994 .25204 L
s
[(0.001)] .13119 .25204 1 0 Mshowa
.14369 .3707 m
.14994 .3707 L
s
[(0.0015)] .13119 .3707 1 0 Mshowa
.14369 .48936 m
.14994 .48936 L
s
[(0.002)] .13119 .48936 1 0 Mshowa
.14369 .60803 m
.14994 .60803 L
s
[(0.0025)] .13119 .60803 1 0 Mshowa
.125 Mabswid
.14369 .03845 m
.14744 .03845 L
s
.14369 .06218 m
.14744 .06218 L
s
.14369 .08591 m
.14744 .08591 L
s
.14369 .10964 m
.14744 .10964 L
s
.14369 .15711 m
.14744 .15711 L
s
.14369 .18084 m
.14744 .18084 L
s
.14369 .20457 m
.14744 .20457 L
s
.14369 .22831 m
.14744 .22831 L
s
.14369 .27577 m
.14744 .27577 L
s
.14369 .2995 m
.14744 .2995 L
s
.14369 .32324 m
.14744 .32324 L
s
.14369 .34697 m
.14744 .34697 L
s
.14369 .39443 m
.14744 .39443 L
s
.14369 .41817 m
.14744 .41817 L
s
.14369 .4419 m
.14744 .4419 L
s
.14369 .46563 m
.14744 .46563 L
s
.14369 .5131 m
.14744 .5131 L
s
.14369 .53683 m
.14744 .53683 L
s
.14369 .56056 m
.14744 .56056 L
s
.14369 .58429 m
.14744 .58429 L
s
.25 Mabswid
.14369 0 m
.14369 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
0 0 1 r
.5 Mabswid
.02381 .60332 m
.02498 .60266 L
.02615 .60199 L
.02848 .60066 L
.03316 .59801 L
.0425 .5927 L
.06277 .58118 L
.08169 .57043 L
.10025 .55989 L
.12037 .54846 L
.13915 .53779 L
.15951 .52622 L
.17949 .51487 L
.19813 .50428 L
.21834 .4928 L
.23721 .48208 L
.25571 .47157 L
.27578 .46017 L
.29451 .44953 L
.31481 .43799 L
.33473 .42667 L
.35332 .41611 L
.37348 .40466 L
.39229 .39397 L
.41268 .38239 L
.4327 .37102 L
.45137 .36041 L
.47162 .34891 L
.49052 .33817 L
.50905 .32764 L
.52915 .31622 L
.54791 .30556 L
.56825 .29401 L
.58821 .28267 L
.60683 .27209 L
.62702 .26062 L
.64587 .24992 L
.66435 .23942 L
.6844 .22803 L
.7031 .2174 L
.72338 .20588 L
.74232 .19513 L
.76088 .18458 L
.78102 .17314 L
.79981 .16246 L
.82018 .15089 L
.84018 .13953 L
.85883 .12894 L
.87905 .11745 L
.89794 .10672 L
Mistroke
.91645 .09621 L
.93653 .0848 L
.95527 .07415 L
.97558 .06261 L
.97619 .06227 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[11]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool004?oo`@000?oo`8000Coo`03003ooooo00;oo`@000Coo`04003oooooool40002ool30004ool4
0003ool20004ool00`00ooooo`02ool40004ool01000oooooooo0`000ooo10000ooo0P001?oo00<0
0?ooool00_oo10001?oo00<00?ooool00_oo00<00?ooool00_oo0`001ooo10000ooo0P001?oo00<0
0?ooool00_oo10000_oo0`001ooo10000ooo0P001?oo00<00?ooool00_oo10000_oo0`000ooo1000
0_oo0`005ooo0015ool01@00oooooooo00000_oo00<00?ooool02?oo00<00?ooool00ooo00D00?oo
ooooo`0000Soo`03003ooooo00?oo`05003oooooool00002ool00`00ooooo`08ool00`00ooooo`03
ool00`00ooooo`04ool00`00ooooo`02ool01@00oooooooo00000_oo00<00?ooool02?oo00<00?oo
ool00ooo00<00?ooool00_oo00<00?ooool01Ooo00<00?ooool01_oo00D00?ooooooo`0000;oo`03
003ooooo00Soo`03003ooooo00Coo`03003ooooo00Koo`05003oooooool00002ool00`00ooooo`08
ool00`00ooooo`04ool01000ooooo`002?oo00<00?ooool05?oo0015ool01@00oooooooo00000_oo
00<00?ooool02?oo00<00?ooool01@000ooo00<00?ooool01Ooo00<00?ooool00ooo00D00?oooooo
o`0000;oo`03003ooooo00Soo`03003ooooo00D000Goo`03003ooooo00;oo`05003oooooool00002
ool00`00ooooo`08ool00`00ooooo`050004ool00`00ooooo`04ool00`00ooooo`06ool01@00oooo
oooo00000_oo00<00?ooool02?oo00<00?ooool01?oo00<00?ooool01_oo00D00?ooooooo`0000;o
o`03003ooooo00Soo`03003ooooo00Coo`05003oooooool00007ool00`00ooooo`0Dool004Goo`04
003oooooool3000;ool01000ooooo`000_oo00<00?ooool00ooo00@00?ooooooo`<000Koo`04003o
ooooool3000;ool01000ooooo`000_oo00@00?ooooooo`<000Goo`04003oooooool3000;ool01000
ooooo`000_oo00<00?ooool00ooo00@00?ooooooo`<000Woo`04003oooooool3000;ool01000oooo
oooo0`002Ooo00@00?ooooooo`<000_oo`04003oooooool30005ool01000oooooooo0`005ooo0015
ool01@00oooooooo00003Ooo00L00?ooooooo`00ool00006ool01000ooooo`002?oo00D00?oooooo
o`0000goo`07003oooooool00?oo00000ooo00<00?ooool01Ooo00D00?ooooooo`0000goo`07003o
ooooool00?oo00001_oo00@00?ooool000_oo`05003oooooool0000=ool01@00oooooooo00002ooo
00D00?ooooooo`0000goo`05003oooooool00008ool01000ooooo`006Ooo0013ool30003ool00`00
ooooo`09ool30004ool20003ool01000ooooo`000_oo00<00?ooool01?oo0`000ooo00<00?ooool0
2Ooo0`001?oo0P000ooo00<00?ooool00ooo0`000ooo00<00?ooool02Ooo0`001?oo0P000ooo00@0
0?ooool000;oo`03003ooooo00Ooo`<000?oo`03003ooooo00Woo`<000?oo`03003ooooo00Ooo`<0
00?oo`03003ooooo00Woo`<000?oo`03003ooooo00?oo`04003ooooo0002ool00`00ooooo`0Gool0
02ooo`03003ooooo01?oo`03003ooooo00;oo`<000[oo`03003ooooo00?oo`03003ooooo00;oo`80
00?oo`@000Goo`03003ooooo00;oo`<000[oo`03003ooooo00?oo`04003oooooool40004ool00`00
ooooo`02ool3000:ool00`00ooooo`03ool01000oooooooo10000_oo10002?oo00<00?ooool00_oo
0`002_oo00@00?ooooooo`@000Soo`03003ooooo00;oo`<000[oo`04003oooooool40003ool20003
ool4000Fool002ooo`03003ooooo0>koo`00;ooo00<00?ooool0k_oo000_ool00`00ooooo`3^ool0
00Soool0010000Woo`003ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?oo
ool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?oo
ool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?oo
ool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?oo
ool01?oo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?oo
ool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?oo
ool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?oo
ool01ooo000_ool00`00ooooo`0Uool00`00ooooo`0Uool00`00ooooo`0Tool00`00ooooo`0Uool0
0`00ooooo`0Uool00`00ooooo`0Wool002ooo`03003ooooo0>koo`00;ooo00<00?ooool0k_oo000_
ool00`00ooooo`3^ool002ooo`800>ooo`00;ooo00<00?ooool0k_oo000_ool00`00ooooo`3^ool0
02ooo`03003ooooo0>koo`00;ooo00<00?ooool0k_oo000_ool00`00ooooo`3^ool002ooo`03003o
oooo0>koo`00;ooo0P00g_oo0P0O3ooo000_ool00`00ooooo`3Kool201lAool002ooo`03003ooooo
0=Woo`807a?oo`00;ooo00<00?ooool0f?oo00<07oooool04ooo000_ool00`00ooooo`3Fool201lF
ool002ooo`03003ooooo0=Coo`807aSoo`00;ooo0P00e?oo00<07oooool06?oo000_ool00`00oooo
o`3Aool201lKool002ooo`03003ooooo0<ooo`807agoo`00;ooo00<00?ooool0c_oo00<07oooool0
7Ooo000_ool00`00ooooo`3<ool201lPool002ooo`03003ooooo0<[oo`807b;oo`00;ooo00<00?oo
ool0bOoo00<07oooool08_oo000_ool20038ool201lUool002ooo`03003ooooo0<Goo`807bOoo`00
;ooo00<00?ooool0`ooo0P0O:Ooo000_ool00`00ooooo`31ool201l[ool000[oo`8000Coo`03003o
oooo00?oo`8000Coo`8000Coo`8000?oo`<000Goo`03003ooooo0;ooo`807bgoo`002Ooo00@00?oo
ool000Soo`04003ooooo0002ool01000ooooo`000_oo00@00?ooool000Goo`03003ooooo00;oo`03
003ooooo0;koo`0301oooooo02goo`002Ooo00@00?ooool000Soo`04003ooooo0002ool01000oooo
o`000_oo00@00?ooool000Goo`03003ooooo00;oo`<00;coo`807c3oo`002Ooo00@00?ooool000So
o`04003ooooo0002ool01000ooooo`000_oo00@00?ooool000;oo`<000Goo`03003ooooo0;[oo`80
7c;oo`002Ooo00@00?ooool000Soo`04003ooooo0002ool01000ooooo`000_oo00@00?ooool000;o
o`03003ooooo00Goo`03003ooooo0;Woo`0301oooooo03;oo`002Ooo00@00?ooool000Soo`04003o
oooo0002ool01000ooooo`000_oo00@00?ooool000;oo`03003ooooo00Goo`03003ooooo0;Ooo`80
7cGoo`002_oo0P002_oo0P001?oo0P001?oo0P000ooo10001?oo00<00?ooool0]Ooo0P0O=ooo000_
ool00`00ooooo`2cool201liool002ooo`800;;oo`807c_oo`00;ooo00<00?ooool0[ooo0P0O?Ooo
000_ool00`00ooooo`2^ool00`0Oooooo`0mool002ooo`03003ooooo0:coo`807d3oo`00;ooo00<0
0?ooool0Z_oo0P0O@_oo000_ool00`00ooooo`2Yool00`0Oooooo`12ool002ooo`03003ooooo0:Oo
o`807dGoo`00;ooo0P00Y_oo0P0OAooo000_ool00`00ooooo`2Tool00`0Oooooo`17ool002ooo`03
003ooooo0:;oo`807d[oo`00;ooo00<00?ooool0X?oo0P0OC?oo000_ool00`00ooooo`2Oool00`0O
ooooo`1<ool002ooo`03003ooooo09goo`807dooo`00;ooo0P00W?oo0P0ODOoo000_ool00`00oooo
o`2Iool201mCool002ooo`03003ooooo09Ooo`807eGoo`00;ooo00<00?ooool0UOoo0P0OEooo000_
ool00`00ooooo`2Cool201mIool002ooo`03003ooooo097oo`807e_oo`00;ooo00<00?ooool0Sooo
0P0OGOoo000_ool2002?ool00`0Oooooo`1Mool002ooo`03003ooooo08coo`807f3oo`00;ooo00<0
0?ooool0Rooo00<07oooool0H?oo000_ool00`00ooooo`29ool201mSool0013oo`8000Coo`03003o
oooo00?oo`8000Coo`8000?oo`@000Coo`03003ooooo08Ooo`807fGoo`003ooo00@00?ooool000So
o`04003ooooo0002ool01000ooooo`001?oo00<00?ooool00ooo00<00?ooool0QOoo0P0OIooo000?
ool01000ooooo`002?oo00@00?ooool000;oo`04003ooooo0004ool00`00ooooo`03ool30023ool2
01mYool000ooo`04003ooooo0008ool01000ooooo`000_oo00@00?ooool000Coo`03003ooooo00?o
o`03003ooooo087oo`807f_oo`003ooo00@00?ooool000Soo`04003ooooo0002ool01000ooooo`00
1?oo00<00?ooool00ooo00<00?ooool0Oooo0P0OKOoo000?ool01000ooooo`002?oo00@00?ooool0
00;oo`04003ooooo0002ool30005ool00`00ooooo`1mool201m_ool0013oo`8000[oo`8000Coo`80
00Goo`03003ooooo00?oo`03003ooooo07_oo`807g7oo`00;ooo00<00?ooool0N_oo00<07oooool0
LOoo000_ool2001iool201mdool002ooo`03003ooooo07Koo`807gKoo`00;ooo00<00?ooool0MOoo
00<07oooool0M_oo000_ool00`00ooooo`1cool201miool002ooo`03003ooooo077oo`807g_oo`00
;ooo00<00?ooool0Kooo0P0OOOoo000_ool00`00ooooo`1]ool201moool002ooo`8006coo`807h7o
o`00;ooo00<00?ooool0J_oo00<07oooool0POoo000_ool00`00ooooo`1Xool201n4ool002ooo`03
003ooooo06Koo`807hKoo`00;ooo00<00?ooool0IOoo00<07oooool0Q_oo000_ool00`00ooooo`1S
ool201n9ool002ooo`8006;oo`807h_oo`00;ooo00<00?ooool0H?oo00<07oooool0Rooo000_ool0
0`00ooooo`1Nool201n>ool002ooo`03003ooooo05coo`807i3oo`00;ooo00<00?ooool0Fooo00<0
7oooool0T?oo000_ool00`00ooooo`1Iool201nCool002ooo`03003ooooo05Ooo`807iGoo`00;ooo
0P00E_oo0P0OUooo000_ool00`00ooooo`1Cool201nIool002ooo`03003ooooo057oo`807i_oo`00
;ooo00<00?ooool0D?oo00<07oooool0Vooo000:ool20004ool00`00ooooo`03ool20004ool20003
ool40002ool30005ool00`00ooooo`1>ool201nNool000Woo`04003ooooo0008ool01000ooooo`00
0_oo00@00?ooool000Coo`03003ooooo00Coo`03003ooooo00;oo`03003ooooo04coo`807j3oo`00
2Ooo00@00?ooool000Soo`04003ooooo0002ool01000ooooo`001?oo00<00?ooool01?oo00<00?oo
ool00_oo0`00Booo00<07oooool0X?oo0009ool01000ooooo`002?oo00@00?ooool000;oo`04003o
oooo0004ool01000oooooooo0`001Ooo00<00?ooool0BOoo0P0OXooo0009ool01000ooooo`002?oo
00@00?ooool000;oo`04003ooooo0004ool01@00oooooooo00001ooo00<00?ooool0Aooo0P0OYOoo
0009ool01000ooooo`002?oo00@00?ooool000;oo`04003ooooo0002ool30003ool00`00ooooo`05
ool00`00ooooo`15ool201nWool000[oo`8000[oo`8000Coo`8000Goo`04003oooooool40004ool0
0`00ooooo`13ool201nYool002ooo`03003ooooo047oo`807j_oo`00;ooo0P00@Ooo00<07oooool0
Zooo000_ool00`00ooooo`0nool201n^ool002ooo`03003ooooo03coo`807k3oo`00;ooo00<00?oo
ool0>ooo00<07oooool0/?oo000_ool00`00ooooo`0iool201ncool002ooo`03003ooooo03Ooo`80
7kGoo`00;ooo00<00?ooool0=_oo00<07oooool0]Ooo000_ool2000eool201nhool002ooo`03003o
oooo03;oo`807k[oo`00;ooo00<00?ooool0<?oo0P0O_?oo000_ool00`00ooooo`0^ool201nnool0
02ooo`03003ooooo02coo`807l3oo`00;ooo00<00?ooool0:ooo00<07oooool0`?oo000_ool2000Z
ool201o3ool002ooo`03003ooooo02Ooo`807lGoo`00;ooo00<00?ooool09_oo00<07oooool0aOoo
000_ool00`00ooooo`0Tool201o8ool002ooo`03003ooooo02;oo`807l[oo`00;ooo00<00?ooool0
8?oo0P0Oc?oo000_ool00`00ooooo`0Nool201o>ool002ooo`8001goo`807m3oo`00;ooo00<00?oo
ool06ooo00<07oooool0d?oo000_ool00`00ooooo`0Iool201oCool002ooo`03003ooooo01Ooo`80
7mGoo`004?oo0P001?oo00<00?ooool00ooo0P001?oo0P000ooo10001?oo00<00?ooool05_oo00<0
7oooool0eOoo000?ool01000ooooo`002?oo00@00?ooool000;oo`04003ooooo0002ool00`00oooo
o`05ool00`00ooooo`0Dool201oHool000ooo`04003ooooo0008ool01000ooooo`000_oo00@00?oo
ool000?oo`03003ooooo00Coo`<001;oo`807m[oo`003ooo00@00?ooool000Soo`04003ooooo0002
ool01000ooooo`001?oo00<00?ooool00ooo00<00?ooool04Ooo00<07oooool0f_oo000?ool01000
ooooo`002?oo00@00?ooool000;oo`04003ooooo0005ool00`00ooooo`02ool00`00ooooo`0?ool2
01oMool000ooo`04003ooooo0008ool01000ooooo`000_oo00@00?ooool000;oo`04003ooooo0004
ool00`00ooooo`0=ool201oOool0013oo`8000[oo`8000Coo`8000Coo`8000Goo`03003ooooo00co
o`0301oooooo0=ooo`00;ooo00<00?ooool02_oo0P0Oh_oo000_ool20009ool201oTool002ooo`03
003ooooo00Koo`807nKoo`00;ooo00<00?ooool01?oo0P0Oj?oo000_ool00`00ooooo`02ool201oZ
ool002ooo`05003oooooool07`3/ool002ooo`04003oo`0O01o]ool002ooo`807nooo`00;_oo00<0
7`000000kooo000/ool201l00ooo003oo`3_ool002[oo`807`?oo`03003ooooo0>koo`00:?oo0P0O
1Ooo00<00?ooool0k_oo000Vool201l7ool00`00ooooo`3^ool002Coo`807`Woo`03003ooooo0>ko
o`008ooo00<07oooool02Ooo0P00kooo000Qool201l<ool00`00ooooo`3^ool001ooo`807`koo`03
003ooooo0>koo`007_oo00<07oooool03_oo00<00?ooool0k_oo000Lool201lAool00`00ooooo`3^
ool001[oo`807a?oo`03003ooooo0>koo`006Ooo00<07oooool04ooo00<00?ooool0k_oo000Gool2
01lFool2003_ool001Goo`807aSoo`03003ooooo0>koo`004ooo0P0O6_oo00<00?ooool0k_oo000A
ool201lLool00`00ooooo`3^ool000[oo`8000?oo`807`Goo`8000Coo`8000?oo`@000;oo`<000Go
o`03003ooooo0>koo`002Ooo00@00?ooool000;oo`0301oooooo00?oo`04003ooooo0002ool01000
ooooo`000_oo00<00?ooool01_oo00<00?ooool00_oo00<00?ooool0k_oo0009ool01000ooooo`00
2?oo00@00?ooool000;oo`04003ooooo0003ool00`00ooooo`05ool00`00ooooo`02ool3003^ool0
00Woo`04003ooooo0008ool01000ooooo`000_oo00@00?ooool000Coo`04003oooooool30005ool0
0`00ooooo`3^ool000Woo`04003ooooo0008ool01000ooooo`000_oo00@00?ooool000Goo`04003o
oooo0007ool00`00ooooo`3^ool000Woo`04003ooooo0008ool01000ooooo`000_oo00@00?ooool0
00;oo`04003ooooo0002ool00`00ooooo`05ool00`00ooooo`3^ool000[oo`8000[oo`8000Coo`80
00Coo`8000?oo`@00?Goo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {16.137, -0.000193231, \
6.28727*^-5, 1.56187*^-5}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[11]="]
}, Open  ]],

Cell["\<\
The linear interpolation method is sufficient for most viewing \
purposes, such as the Poincar\[EAcute]  section examples shown in the \
following section.   Note that for Boolean valued event functions, linear \
interpolation is effectively only one bisection step, so the linear \
interpolation method may be inadequate for graphics.  \
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Brent's Method", "Subsection"],

Cell[TextData[{
  "The default location method is the event location method ",
  StyleBox["\"Brent\"", "MR"],
  ", finding the location of the event using ",
  StyleBox["FindRoot", "MR"],
  " with ",
  StyleBox[Cell[BoxData[
      ButtonBox["Brent",
        ButtonFunction:>(FrontEndExecute[ {
            FrontEnd`NotebookLocate[ #2], 
            FrontEnd`HelpBrowserLookup[ 
            "RefGuide", 
              "Advanced Documentation: Unconstrained Optimization"]}]&),
        ButtonData:>{
          
          FrontEnd`FileName[ {$TopDirectory, "Documentation", "English", 
            "RefGuide", "AdvancedDocumentation", "Optimization"}, 
            "Unconstrained.nb"], "t:13"},
        Active->True,
        ButtonStyle->"RefGuideLinkText",
        ButtonNote->"Advanced Documentation"]], "Text"], "MR"],
  "'s method.  Brent's method starts with a bracketed root and combines steps \
based on interpolation and bisection, guaranteeing a convergence rate at \
least as good as bisection.  You can control the accuracy and precision to \
which ",
  StyleBox["FindRoot", "MR"],
  " tries to get the root of the event function using method options for the \
",
  StyleBox["\"Brent\"", "MR"],
  " event location method.  The default is to find the root to the same \
accuracy and precision as ",
  StyleBox["NDSolve", "MR"],
  " is using for local error control."
}], "Text"],

Cell[TextData[{
  "For methods which support continuous or dense output, the argument for the \
event function can be found quite efficiently simply by using the continuous \
output formula.  However, for methods which do not support continuous output, \
 the solution needs to be computed by taking a step of the underlying method, \
which can be relatively expensive.  An alternate way of getting a solution \
approximation which is not accurate to the method order, but is consistent \
with using FindRoot on the InterpolatingFunction object returned from ",
  StyleBox["NDSolve", "MR"],
  " is to use cubic Hermite interpolation, obtaining approximate solution \
values in the middle of the step by interpolation based on the solution \
values and solution derivative values at the step ends."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Comparison", "Subsection"],

Cell["\<\
This example integrates the pendulum equation for a number of \
different event location methods and compares the time when the event is \
found.\
\>", "Text"],

Cell["This defines the event location methods to use.", "MathCaption"],

Cell[BoxData[
    \(\(eventmethods\  = \ {"\<StepBegin\>", "\<StepEnd\>", \
"\<LinearInterpolation\>", Automatic};\)\)], "Input"],

Cell["\<\
This integrates the system and prints out the method used and the \
value of the independent variable when the integration is terminated.\
\>", \
"MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Map[\[IndentingNewLine]NDSolve[{\(x''\)[t]\  + \ 
                  Sin[x[t]]\  \[Equal] \ 0, \ 
              x[0]\  \[Equal] \ 3, \ \(x'\)[0]\  \[Equal] \ 0}, 
            x, {t, 0, \[Infinity]}, \ 
            Method \[Rule] {"\<EventLocator\>", \[IndentingNewLine]\ "\<Event\
\>" \[Rule] \(x'\)[
                    t], \ \[IndentingNewLine]"\<EventAction\>" \[RuleDelayed] \
\ Throw[Print[#, "\<: t = \>", 
                      t], \ "\<StopIntegration\>"], "\<Direction\>" \[Rule] \
\(-1\), Method -> "\<ExplicitRungeKutta\>", \ \
\[IndentingNewLine]"\<EventLocationMethod\>" \[Rule] #}] &, \
\[IndentingNewLine]eventmethods\[IndentingNewLine]];\)\)], "Input"],

Cell[BoxData[
    InterpretationBox[\("StepBegin"\[InvisibleSpace]": t = \
"\[InvisibleSpace]16.06829641532804`\),
      SequenceForm[ "StepBegin", ": t = ", 16.068296415328039],
      Editable->False]], "Print",
  CellLabel->"From In[26]:="],

Cell[BoxData[
    InterpretationBox[\("StepEnd"\[InvisibleSpace]": t = \
"\[InvisibleSpace]16.51798788405511`\),
      SequenceForm[ "StepEnd", ": t = ", 16.51798788405511],
      Editable->False]], "Print",
  CellLabel->"From In[26]:="],

Cell[BoxData[
    InterpretationBox[\("LinearInterpolation"\[InvisibleSpace]": t = "\
\[InvisibleSpace]16.154120673095942`\),
      SequenceForm[ "LinearInterpolation", ": t = ", 16.154120673095942],
      Editable->False]], "Print",
  CellLabel->"From In[26]:="],

Cell[BoxData[
    InterpretationBox[\(Automatic\[InvisibleSpace]": t = \
"\[InvisibleSpace]16.155540526183064`\),
      SequenceForm[ Automatic, ": t = ", 16.155540526183064],
      Editable->False]], "Print",
  CellLabel->"From In[26]:="]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
  CellTags->"s:3"],

Cell[CellGroupData[{

Cell["Falling Body", "Subsection"],

Cell[TextData[{
  "This system models a body falling under the force of gravity encountering \
air resistance (see [",
  ButtonBox["M04",
    ButtonData:>{"References.nb", "M04"},
    ButtonStyle->"Hyperlink"],
  "])."
}], "Text"],

Cell["\<\
The event action stores the time when the falling body hits the \
ground and stops the integration.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol = 
      y[t] /. First[
          NDSolve[{\(y''\)[t] \[Equal] \(-1\) + \(y'\)[t]^2, 
              y[0] \[Equal] 1, \(y'\)[0] \[Equal] 0}, y, {t, 0, Infinity}, 
            Method \[Rule] {"\<EventLocator\>", "\<Event\>" \[RuleDelayed] 
                  y[t], "\<EventAction\>" \[RuleDelayed] 
                  Throw[tend = t, "\<StopIntegration\>"]}]]\)], "Input",
  CellLabel->"In[11]:="],

Cell[BoxData[
    RowBox[{
      TagBox[\(InterpolatingFunction[{{0.`, 1.657454454759687`}}, "<>"]\),
        False,
        Editable->False], "[", "t", "]"}]], "Output",
  CellLabel->"Out[11]="]
}, Open  ]],

Cell["\<\
This plots the solution and highlights the initial and final points \
(green and red) by encircling them.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(plt = 
        Plot[sol, {t, 0, tend}, Frame \[Rule] True, Axes \[Rule] False, 
          PlotStyle \[Rule] Blue, 
          DisplayFunction \[Rule] Identity];\)\[IndentingNewLine]\), "\n", 
    \(\(grp = 
        Graphics[{{Green, Circle[{0, 1}, 0.025]}, {Red, 
              Circle[{tend, sol /. t \[Rule] tend}, 
                0.025]}}];\)\n\), "\[IndentingNewLine]", 
    \(Show[plt, grp, DisplayFunction \[Rule] $DisplayFunction]\)}], "Input",
  CellLabel->"In[15]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.037754 0.557778 0.0287295 0.560575 [
[.03775 -0.0125 -3 -9 ]
[.03775 -0.0125 3 0 ]
[.1772 -0.0125 -12 -9 ]
[.1772 -0.0125 12 0 ]
[.31664 -0.0125 -9 -9 ]
[.31664 -0.0125 9 0 ]
[.45609 -0.0125 -12 -9 ]
[.45609 -0.0125 12 0 ]
[.59553 -0.0125 -3 -9 ]
[.59553 -0.0125 3 0 ]
[.73498 -0.0125 -12 -9 ]
[.73498 -0.0125 12 0 ]
[.87442 -0.0125 -9 -9 ]
[.87442 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .02873 -6 -4.5 ]
[-0.0125 .02873 0 4.5 ]
[-0.0125 .14084 -18 -4.5 ]
[-0.0125 .14084 0 4.5 ]
[-0.0125 .25296 -18 -4.5 ]
[-0.0125 .25296 0 4.5 ]
[-0.0125 .36507 -18 -4.5 ]
[-0.0125 .36507 0 4.5 ]
[-0.0125 .47719 -18 -4.5 ]
[-0.0125 .47719 0 4.5 ]
[-0.0125 .5893 -6 -4.5 ]
[-0.0125 .5893 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03775 0 m
.03775 .00625 L
s
[(0)] .03775 -0.0125 0 1 Mshowa
.1772 0 m
.1772 .00625 L
s
[(0.25)] .1772 -0.0125 0 1 Mshowa
.31664 0 m
.31664 .00625 L
s
[(0.5)] .31664 -0.0125 0 1 Mshowa
.45609 0 m
.45609 .00625 L
s
[(0.75)] .45609 -0.0125 0 1 Mshowa
.59553 0 m
.59553 .00625 L
s
[(1)] .59553 -0.0125 0 1 Mshowa
.73498 0 m
.73498 .00625 L
s
[(1.25)] .73498 -0.0125 0 1 Mshowa
.87442 0 m
.87442 .00625 L
s
[(1.5)] .87442 -0.0125 0 1 Mshowa
.125 Mabswid
.06564 0 m
.06564 .00375 L
s
.09353 0 m
.09353 .00375 L
s
.12142 0 m
.12142 .00375 L
s
.14931 0 m
.14931 .00375 L
s
.20509 0 m
.20509 .00375 L
s
.23298 0 m
.23298 .00375 L
s
.26087 0 m
.26087 .00375 L
s
.28875 0 m
.28875 .00375 L
s
.34453 0 m
.34453 .00375 L
s
.37242 0 m
.37242 .00375 L
s
.40031 0 m
.40031 .00375 L
s
.4282 0 m
.4282 .00375 L
s
.48398 0 m
.48398 .00375 L
s
.51187 0 m
.51187 .00375 L
s
.53975 0 m
.53975 .00375 L
s
.56764 0 m
.56764 .00375 L
s
.62342 0 m
.62342 .00375 L
s
.65131 0 m
.65131 .00375 L
s
.6792 0 m
.6792 .00375 L
s
.70709 0 m
.70709 .00375 L
s
.76287 0 m
.76287 .00375 L
s
.79075 0 m
.79075 .00375 L
s
.81864 0 m
.81864 .00375 L
s
.84653 0 m
.84653 .00375 L
s
.00987 0 m
.00987 .00375 L
s
.90231 0 m
.90231 .00375 L
s
.9302 0 m
.9302 .00375 L
s
.95809 0 m
.95809 .00375 L
s
.98598 0 m
.98598 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .02873 m
.00625 .02873 L
s
[(0)] -0.0125 .02873 1 0 Mshowa
0 .14084 m
.00625 .14084 L
s
[(0.2)] -0.0125 .14084 1 0 Mshowa
0 .25296 m
.00625 .25296 L
s
[(0.4)] -0.0125 .25296 1 0 Mshowa
0 .36507 m
.00625 .36507 L
s
[(0.6)] -0.0125 .36507 1 0 Mshowa
0 .47719 m
.00625 .47719 L
s
[(0.8)] -0.0125 .47719 1 0 Mshowa
0 .5893 m
.00625 .5893 L
s
[(1)] -0.0125 .5893 1 0 Mshowa
.125 Mabswid
0 .05676 m
.00375 .05676 L
s
0 .08479 m
.00375 .08479 L
s
0 .11282 m
.00375 .11282 L
s
0 .16887 m
.00375 .16887 L
s
0 .1969 m
.00375 .1969 L
s
0 .22493 m
.00375 .22493 L
s
0 .28099 m
.00375 .28099 L
s
0 .30902 m
.00375 .30902 L
s
0 .33705 m
.00375 .33705 L
s
0 .3931 m
.00375 .3931 L
s
0 .42113 m
.00375 .42113 L
s
0 .44916 m
.00375 .44916 L
s
0 .50522 m
.00375 .50522 L
s
0 .53325 m
.00375 .53325 L
s
0 .56128 m
.00375 .56128 L
s
0 .0007 m
.00375 .0007 L
s
0 .61733 m
.00375 .61733 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.03775 .61178 m
.03775 .61803 L
s
.1772 .61178 m
.1772 .61803 L
s
.31664 .61178 m
.31664 .61803 L
s
.45609 .61178 m
.45609 .61803 L
s
.59553 .61178 m
.59553 .61803 L
s
.73498 .61178 m
.73498 .61803 L
s
.87442 .61178 m
.87442 .61803 L
s
.125 Mabswid
.06564 .61428 m
.06564 .61803 L
s
.09353 .61428 m
.09353 .61803 L
s
.12142 .61428 m
.12142 .61803 L
s
.14931 .61428 m
.14931 .61803 L
s
.20509 .61428 m
.20509 .61803 L
s
.23298 .61428 m
.23298 .61803 L
s
.26087 .61428 m
.26087 .61803 L
s
.28875 .61428 m
.28875 .61803 L
s
.34453 .61428 m
.34453 .61803 L
s
.37242 .61428 m
.37242 .61803 L
s
.40031 .61428 m
.40031 .61803 L
s
.4282 .61428 m
.4282 .61803 L
s
.48398 .61428 m
.48398 .61803 L
s
.51187 .61428 m
.51187 .61803 L
s
.53975 .61428 m
.53975 .61803 L
s
.56764 .61428 m
.56764 .61803 L
s
.62342 .61428 m
.62342 .61803 L
s
.65131 .61428 m
.65131 .61803 L
s
.6792 .61428 m
.6792 .61803 L
s
.70709 .61428 m
.70709 .61803 L
s
.76287 .61428 m
.76287 .61803 L
s
.79075 .61428 m
.79075 .61803 L
s
.81864 .61428 m
.81864 .61803 L
s
.84653 .61428 m
.84653 .61803 L
s
.00987 .61428 m
.00987 .61803 L
s
.90231 .61428 m
.90231 .61803 L
s
.9302 .61428 m
.9302 .61803 L
s
.95809 .61428 m
.95809 .61803 L
s
.98598 .61428 m
.98598 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .02873 m
1 .02873 L
s
.99375 .14084 m
1 .14084 L
s
.99375 .25296 m
1 .25296 L
s
.99375 .36507 m
1 .36507 L
s
.99375 .47719 m
1 .47719 L
s
.99375 .5893 m
1 .5893 L
s
.125 Mabswid
.99625 .05676 m
1 .05676 L
s
.99625 .08479 m
1 .08479 L
s
.99625 .11282 m
1 .11282 L
s
.99625 .16887 m
1 .16887 L
s
.99625 .1969 m
1 .1969 L
s
.99625 .22493 m
1 .22493 L
s
.99625 .28099 m
1 .28099 L
s
.99625 .30902 m
1 .30902 L
s
.99625 .33705 m
1 .33705 L
s
.99625 .3931 m
1 .3931 L
s
.99625 .42113 m
1 .42113 L
s
.99625 .44916 m
1 .44916 L
s
.99625 .50522 m
1 .50522 L
s
.99625 .53325 m
1 .53325 L
s
.99625 .56128 m
1 .56128 L
s
.99625 .0007 m
1 .0007 L
s
.99625 .61733 m
1 .61733 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
0 0 1 r
.5 Mabswid
.03775 .5893 m
.03889 .5893 L
.04002 .5893 L
.04116 .58929 L
.04229 .58929 L
.04342 .58928 L
.04456 .58926 L
.04683 .58923 L
.04796 .58921 L
.0491 .58919 L
.05136 .58914 L
.0559 .58901 L
.05836 .58892 L
.06082 .58883 L
.06574 .5886 L
.07066 .58833 L
.07557 .58802 L
.08476 .58732 L
.09394 .58646 L
.11195 .58436 L
.13149 .58143 L
.14972 .57809 L
.16948 .57382 L
.18887 .56898 L
.20697 .56389 L
.22659 .55778 L
.2449 .5515 L
.26286 .54484 L
.28234 .53705 L
.30052 .52927 L
.32023 .5203 L
.33957 .51095 L
.35761 .50178 L
.37718 .49135 L
.39545 .48117 L
.41524 .46968 L
.43467 .45795 L
.45279 .44662 L
.47245 .43394 L
.4908 .42174 L
.50878 .40946 L
.5283 .39578 L
.54651 .38271 L
.56625 .36822 L
.58563 .35368 L
.6037 .33986 L
.6233 .3246 L
.6416 .31011 L
.65953 .29569 L
.679 .27981 L
Mistroke
.69716 .26479 L
.71684 .2483 L
.73522 .23271 L
.75324 .21727 L
.77279 .20034 L
.79104 .18438 L
.81081 .16693 L
.83022 .14965 L
.84832 .13339 L
.86795 .11564 L
.88628 .09895 L
.90425 .08249 L
.92375 .06452 L
.94194 .04766 L
.96166 .02928 L
.96225 .02873 L
Mfstroke
0 1 0 r
newpath
matrix currentmatrix
0.0139445 0.0140144 scale
2.70717 42.0497 1 0 365.73 arc
setmatrix s
1 0 0 r
newpath
matrix currentmatrix
0.0139445 0.0140144 scale
69.0059 2.05004 1 0 365.73 arc
setmatrix s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[15]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool001koo`8001[oo`8000Coo`03003ooooo00;oo`@000;oo`<001Coo`8000Coo`03003ooooo00;o
o`<001Coo`8000Coo`03003ooooo00?oo`03003ooooo00;oo`<001Woo`@001Soo`@000?oo`03003o
oooo00;oo`@000;oo`<001?oo`@000?oo`03003ooooo00;oo`<001coo`007Ooo00@00?ooool001So
o`04003ooooo0008ool00`00ooooo`06ool00`00ooooo`0@ool01000ooooo`002ooo00<00?ooool0
4?oo00@00?ooool000Woo`03003ooooo00Goo`03003ooooo01Soo`03003ooooo01Woo`03003ooooo
00Ooo`03003ooooo00Koo`03003ooooo01;oo`03003ooooo00[oo`03003ooooo01Woo`007Ooo00@0
0?ooool001Soo`04003ooooo0009ool00`00ooooo`05ool00`00ooooo`0@ool01000ooooo`002ooo
00<00?ooool04?oo00@00?ooool000[oo`03003ooooo00Coo`03003ooooo01Soo`03003ooooo01Wo
o`03003ooooo00Soo`03003ooooo00Goo`03003ooooo01;oo`03003ooooo00[oo`03003ooooo01Wo
o`007Ooo00@00?ooool001Soo`04003ooooo000:ool01000oooooooo0`004ooo00@00?ooool000So
o`<001?oo`04003ooooo000:ool01000oooooooo0`006ooo00<00?ooool06Ooo00<00?ooool02Ooo
00@00?ooooooo`<001Goo`03003ooooo00Ooo`<001coo`007Ooo00@00?ooool001Soo`04003ooooo
000;ool01000ooooo`005Ooo00@00?ooool000Soo`03003ooooo01?oo`04003ooooo000;ool01000
ooooo`007Ooo00<00?ooool06Ooo00<00?ooool02_oo00@00?ooool001Ooo`03003ooooo00Ooo`03
003ooooo01coo`007Ooo00@00?ooool001Soo`04003ooooo0008ool01000ooooo`000_oo00<00?oo
ool04ooo00@00?ooool000Soo`03003ooooo01?oo`04003ooooo0008ool01000ooooo`000_oo00<0
0?ooool06Ooo0`006Ooo0`002Ooo00@00?ooool000;oo`03003ooooo01?oo`<000Woo`03003ooooo
01coo`007_oo0P006_oo0P002_oo0P000ooo10004ooo0P002Ooo10004ooo0P002Ooo10000_oo1000
6_oo00<00?ooool06Ooo00<00?ooool02?oo0P000ooo10005?oo00<00?ooool01ooo10006ooo003o
oolQool00?ooob7oo`00oooo8Ooo000Eoooo000;0001ool001Goo`03003oo`0000Ooo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo00Goo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Goo`04003ooooo0001ool001Goo`03
003ooooo00Ooo`03003ooooo02;oo`03003ooooo02;oo`03003ooooo02;oo`03003ooooo02;oo`03
003ooooo02;oo`03003ooooo02;oo`03003ooooo01koo`40007oo`005Ooo00<00?ooool0oooo1ooo
0@000Ooo000Eool00`00ooooo`3lool2n008ool10001ool001Goo`03003ooooo0?[oo`;h00;oo`;h
00Koo`40007oo`003_oo0P001Ooo00<00?ooool0n_oo00?h0?ooool00_oo00?h0?ooool01?oo0@00
0Ooo000=ool01000ooooo`001?oo00<00?ooool0nOoo00?h0?ooool01?oo00?h0?ooool00ooo0@00
0Ooo000=ool01000ooooo`001?oo0P00n_oo00Ch0?ooool07`?oo`03n03ooooo00;oo`80007oo`00
3Ooo00@00?ooool000Coo`03003ooooo0?Woo`03n03oo`0O00Coo`03n03ooooo00?oo`40007oo`00
3Ooo00@00?ooool000Coo`03003ooooo0?[oo`04n03oooooool2n006ool10001ool000goo`04003o
oooo0004ool00`00ooooo`3iool00`0Ooooh0002n008ool10001ool000koo`8000Goo`03003ooooo
0?Soo`0301oooooo00_oo`40007oo`005Ooo00<00?ooool0mooo00<07oooool03?oo0@000Ooo000E
ool00`00ooooo`3fool00`0Oooooo`0=ool10001ool001Goo`800?Goo`807`ooo`80007oo`005Ooo
00<00?ooool0looo00<07oooool04?oo0@000Ooo000Eool00`00ooooo`3bool00`0Oooooo`0Aool1
0001ool001Goo`03003ooooo0?7oo`0301oooooo01;oo`40007oo`005Ooo00<00?ooool0l?oo00<0
7oooool04ooo0@000Ooo000Eool00`00ooooo`3_ool00`0Oooooo`0Dool10001ool001Goo`03003o
oooo0>koo`0301oooooo01Goo`40007oo`005Ooo00<00?ooool0kOoo00<07oooool05_oo0@000Ooo
000Eool2003]ool00`0Oooooo`0Fool20001ool001Goo`03003ooooo0>[oo`807a[oo`40007oo`00
5Ooo00<00?ooool0jOoo00<07oooool06_oo0@000Ooo000Eool00`00ooooo`3Xool00`0Oooooo`0K
ool10001ool001Goo`03003ooooo0>Ooo`0301oooooo01coo`40007oo`005Ooo00<00?ooool0i_oo
00<07oooool07Ooo0@000Ooo000Eool00`00ooooo`3Uool00`0Oooooo`0Nool10001ool001Goo`80
0>Goo`0301oooooo01koo`80007oo`005Ooo00<00?ooool0hooo00<07oooool08?oo0@000Ooo000E
ool00`00ooooo`3Rool00`0Oooooo`0Qool10001ool001Goo`03003ooooo0>7oo`0301oooooo02;o
o`40007oo`005Ooo00<00?ooool0h?oo00<07oooool08ooo0@000Ooo000Eool00`00ooooo`3Oool0
0`0Oooooo`0Tool10001ool000;oo`8000Coo`03003ooooo00;oo`@000Coo`03003ooooo0=koo`03
01oooooo02Goo`40007oo`0000Goo`00ooooo`0000Soo`03003ooooo00Goo`03003ooooo0=goo`03
01oooooo02Koo`40007oo`0000Goo`00ooooo`0000Woo`03003ooooo00Coo`800=coo`807bSoo`80
007oo`0000Goo`00ooooo`0000[oo`03003ooooo00?oo`03003ooooo0=[oo`0301oooooo02Woo`40
007oo`0000Goo`00ooooo`0000_oo`03003ooooo00;oo`03003ooooo0=Woo`0301oooooo02[oo`40
007oo`0000Goo`00ooooo`0000Soo`04003ooooo0004ool00`00ooooo`3Hool00`0Oooooo`0[ool1
0001ool000;oo`8000[oo`8000Goo`03003ooooo0=Koo`807bkoo`40007oo`005Ooo00<00?ooool0
eOoo00<07oooool0;_oo0@000Ooo000Eool00`00ooooo`3Dool00`0Oooooo`0_ool10001ool001Go
o`800=Coo`0301oooooo02ooo`80007oo`005Ooo00<00?ooool0d_oo00<07oooool0<Ooo0@000Ooo
000Eool00`00ooooo`3Aool00`0Oooooo`0bool10001ool001Goo`03003ooooo0=3oo`0301oooooo
03?oo`40007oo`005Ooo00<00?ooool0cooo00<07oooool0=?oo0@000Ooo000Eool00`00ooooo`3>
ool00`0Oooooo`0eool10001ool001Goo`03003ooooo0<coo`807cSoo`40007oo`005Ooo0P00c?oo
00<07oooool0=ooo0P000Ooo000Eool00`00ooooo`3:ool00`0Oooooo`0iool10001ool001Goo`03
003ooooo0<Woo`0301oooooo03[oo`40007oo`005Ooo00<00?ooool0b?oo00<07oooool0>ooo0@00
0Ooo000Eool00`00ooooo`37ool00`0Oooooo`0lool10001ool001Goo`03003ooooo0<Koo`0301oo
oooo03goo`40007oo`005Ooo00<00?ooool0aOoo00<07oooool0?_oo0@000Ooo000Eool00`00oooo
o`34ool00`0Oooooo`0oool10001ool001Goo`800<?oo`807d7oo`80007oo`005Ooo00<00?ooool0
`Ooo00<07oooool0@_oo0@000Ooo000Eool00`00ooooo`30ool00`0Oooooo`13ool10001ool001Go
o`03003ooooo0;ooo`0301oooooo04Coo`40007oo`005Ooo00<00?ooool0_Ooo0P0OAooo0@000Ooo
0002ool20004ool00`00ooooo`04ool00`00ooooo`03ool00`00ooooo`2lool00`0Oooooo`17ool1
0001ool00005ool00?ooool0000:ool00`00ooooo`03ool00`00ooooo`2kool00`0Oooooo`18ool1
0001ool00005ool00?ooool00007ool50004ool2002kool00`0Oooooo`18ool20001ool00005ool0
0?ooool00007ool01000ooooo`001Ooo00<00?ooool0^?oo0P0OC?oo0@000Ooo00001Ooo003ooooo
00002?oo00<00?oo00001Ooo00<00?ooool0]ooo00<07oooool0C?oo0@000Ooo00001Ooo003ooooo
00002Ooo0P001Ooo00<00?ooool0]_oo00<07oooool0COoo0@000Ooo0002ool2000;ool00`00oooo
o`03ool00`00ooooo`2eool00`0Oooooo`1>ool10001ool001Goo`03003ooooo0;?oo`807e7oo`40
007oo`005Ooo00<00?ooool0/_oo00<07oooool0DOoo0@000Ooo000Eool00`00ooooo`2aool00`0O
ooooo`1Bool10001ool001Goo`800;7oo`0301oooooo05;oo`80007oo`005Ooo00<00?ooool0[ooo
00<07oooool0E?oo0@000Ooo000Eool00`00ooooo`2^ool00`0Oooooo`1Eool10001ool001Goo`03
003ooooo0:goo`0301oooooo05Koo`40007oo`005Ooo00<00?ooool0[?oo00<07oooool0Eooo0@00
0Ooo000Eool00`00ooooo`2Zool201mJool10001ool001Goo`03003ooooo0:Soo`807ecoo`40007o
o`005Ooo0P00Z?oo00<07oooool0Fooo0P000Ooo000Eool00`00ooooo`2Vool00`0Oooooo`1Mool1
0001ool001Goo`03003ooooo0:Coo`807f3oo`40007oo`005Ooo00<00?ooool0Xooo00<07oooool0
H?oo0@000Ooo000Eool00`00ooooo`2Rool00`0Oooooo`1Qool10001ool001Goo`03003ooooo0:7o
o`0301oooooo06;oo`40007oo`005Ooo00<00?ooool0Wooo0P0OIOoo0@000Ooo000Eool00`00oooo
o`2Nool00`0Oooooo`1Uool10001ool001Goo`8009koo`0301oooooo06Goo`80007oo`005Ooo00<0
0?ooool0W?oo00<07oooool0Iooo0@000Ooo000Eool00`00ooooo`2Jool201mZool10001ool001Go
o`03003ooooo09Woo`0301oooooo06[oo`40007oo`005Ooo00<00?ooool0V?oo00<07oooool0Jooo
0@000Ooo0002ool20004ool00`00ooooo`03ool20005ool00`00ooooo`2Gool00`0Oooooo`1/ool1
0001ool00005ool00?ooool00008ool01000ooooo`001?oo00<00?ooool0UOoo0P0OKooo0@000Ooo
00001Ooo003ooooo00002?oo00@00?ooool000Coo`8009Goo`0301oooooo06koo`80007oo`0000Go
o`00ooooo`0000Soo`<000Goo`03003ooooo09?oo`0301oooooo073oo`40007oo`0000Goo`00oooo
o`0000Soo`03003ooooo00Goo`03003ooooo09;oo`0301oooooo077oo`40007oo`0000Goo`00oooo
o`0000Soo`03003ooooo00Goo`03003ooooo093oo`807gCoo`40007oo`000_oo0P002_oo0`001?oo
00<00?ooool0Sooo00<07oooool0M?oo0@000Ooo000Eool00`00ooooo`2>ool00`0Oooooo`1eool1
0001ool001Goo`03003ooooo08coo`807gSoo`40007oo`005Ooo00<00?ooool0R_oo0P0ON_oo0@00
0Ooo000Eool2002:ool00`0Oooooo`1iool20001ool001Goo`03003ooooo08Soo`0301oooooo07_o
o`40007oo`005Ooo00<00?ooool0Q_oo0P0OO_oo0@000Ooo000Eool00`00ooooo`25ool00`0Ooooo
o`1nool10001ool001Goo`03003ooooo08Coo`0301oooooo07ooo`40007oo`005Ooo00<00?ooool0
P_oo0P0OP_oo0@000Ooo000Eool00`00ooooo`20ool201n4ool10001ool001Goo`80083oo`0301oo
oooo08?oo`80007oo`005Ooo00<00?ooool0OOoo0P0OQooo0@000Ooo000Eool00`00ooooo`1kool2
01n9ool10001ool001Goo`03003ooooo07[oo`0301oooooo08Woo`40007oo`005Ooo00<00?ooool0
NOoo00<07oooool0R_oo0@000Ooo000Eool00`00ooooo`1gool201n=ool10001ool001Goo`03003o
oooo07Koo`0301oooooo08goo`40007oo`005Ooo0P00M_oo00<07oooool0SOoo0P000Ooo000Eool0
0`00ooooo`1cool201nAool10001ool001Goo`03003ooooo077oo`807i?oo`40007oo`005Ooo00<0
0?ooool0L?oo00<07oooool0Tooo0@000Ooo000Eool00`00ooooo`1^ool201nFool10001ool001Go
o`03003ooooo06coo`807iSoo`40007oo`000_oo0P001?oo00<00?ooool00ooo0P001Ooo00<00?oo
ool0Jooo00<07oooool0V?oo0@000Ooo00001Ooo003ooooo00002?oo00@00?ooool000Coo`03003o
oooo06Woo`807i_oo`40007oo`0000Goo`00ooooo`0000Soo`04003ooooo0004ool2001Xool201nL
ool20001ool00005ool00?ooool00009ool20005ool00`00ooooo`1Vool00`0Oooooo`2Mool10001
ool00005ool00?ooool00008ool01000ooooo`001?oo00<00?ooool0I?oo0P0OX?oo0@000Ooo0000
1Ooo003ooooo00002?oo00@00?ooool000Coo`03003ooooo06;oo`807j;oo`40007oo`000_oo0P00
2_oo0P001Ooo00<00?ooool0H?oo0P0OY?oo0@000Ooo000Eool00`00ooooo`1Nool201nVool10001
ool001Goo`03003ooooo05coo`807jSoo`40007oo`005Ooo0P00Fooo0P0OZOoo0P000Ooo000Eool0
0`00ooooo`1Hool201n/ool10001ool001Goo`03003ooooo05Koo`807jkoo`40007oo`005Ooo00<0
0?ooool0E?oo0P0O/?oo0@000Ooo000Eool00`00ooooo`1Bool201nbool10001ool001Goo`03003o
oooo053oo`807kCoo`40007oo`005Ooo00<00?ooool0C_oo0P0O]_oo0@000Ooo000Eool00`00oooo
o`1<ool201nhool10001ool001Goo`8004_oo`807kWoo`80007oo`005Ooo00<00?ooool0Aooo0`0O
_?oo0@000Ooo000Eool00`00ooooo`15ool201noool10001ool001Goo`03003ooooo04;oo`<07l7o
o`40007oo`005Ooo00<00?ooool0@?oo0P0Oa?oo0@000Ooo000Eool00`00ooooo`0lool401o6ool1
0001ool001Goo`03003ooooo03Soo`@07l[oo`40007oo`005Ooo0P00=ooo0P0OcOoo0P000Ooo000E
ool00`00ooooo`0bool401o@ool10001ool001Goo`03003ooooo02koo`@07mCoo`40007oo`005Ooo
00<00?ooool0;?oo0P0Of?oo0@000Ooo000Eool00`00ooooo`06ool31n0Oool401oJool10001ool0
01Goo`03003ooooo00Coo`87h0?oo`87h1Soo`D07mkoo`40007oo`003Ooo10001?oo00<00?ooool0
1?oo00<7h?ooool00ooo00<7h?ooool03?oo2P0Ohooo0@000Ooo000?ool00`00ooooo`03ool00`00
ooooo`03ool00`OPooooo`05ool00`OPool07`0;01o]ool10001ool000ooo`03003ooooo00?oo`80
00Coo`041n3oooooool401l00`OP01ooo`3gool20001ool000ooo`03003ooooo00?oo`03003ooooo
00Coo`031n3ooooo00?oo`031n3ooooo0?Woo`40007oo`003ooo00<00?ooool00ooo00<00?ooool0
1?oo0POP0_oo0POPo?oo0@000Ooo000=ool30005ool00`00ooooo`06ool21n3nool10001ool000oo
o`03003ooooo00?oo`03003ooooo0?ooo`Ooo`40007oo`005Ooo00<00?ooool0oooo1ooo0@000Ooo
000Eool00`00ool00007ool00`00ooooo`04ool00`00ooooo`05ool00`00ooooo`04ool00`00oooo
o`04ool00`00ooooo`05ool00`00ooooo`04ool00`00ooooo`05ool00`00ooooo`04ool00`00oooo
o`05ool00`00ooooo`04ool00`00ooooo`04ool00`00ooooo`05ool00`00ooooo`04ool00`00oooo
o`05ool00`00ooooo`04ool00`00ooooo`04ool00`00ooooo`05ool00`00ooooo`04ool00`00oooo
o`05ool00`00ooooo`04ool00`00ooooo`04ool00`00ooooo`05ool00`00ooooo`04ool00`00oooo
o`05ool00`00ooooo`04ool00`00ooooo`04ool00`00ooooo`05ool00`00ooooo`04ool00`00oooo
o`05ool00`00ooooo`04ool00`00ooooo`04ool00`00ooooo`05ool00`00ooooo`04ool00`00oooo
o`05ool01000ooooo`000Ooo000Eoooo000;0001ool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.211629, -0.135681, \
0.0067513, 0.00671762}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[17]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Period of the Van der Pol Oscillator", "Subsection"],

Cell[TextData[{
  "The Van der Pol oscillator is an example of an extremely stiff system of \
ODEs.  The event locator method can call any method for actually doing the \
integration of the ODE system. The default method, ",
  StyleBox["Automatic", "MR"],
  ", automatically switches to a method appropriate for stiff systems when \
necessary, so that stiffness does not present a problem."
}], "Text"],

Cell[TextData[{
  "This integrates the Van der Pol system for a particular value of the \
parameter \[Mu] = 1000 up to the point where the variable ",
  Cell[BoxData[
      \(TraditionalForm\`y\_2\)]],
  " reaches its initial value and direction."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"vsol", " ", "=", " ", 
      RowBox[{"NDSolve", "[", 
        RowBox[{
          RowBox[{"(", GridBox[{
                {\(\(\(y\_1'\)[t]\)\(\ \)\(\[Equal]\)\(\ \)\(y\_2[
                      t]\)\(\ \)\), \(y\_1[0] \[Equal] 2\)},
                {\(\(y\_2'\)[t] \[Equal] 
                    1000\ \((1\  - \ y\_1[t]^2)\)\ y\_2[t]\  - \ 
                      y\_1[t]\), \(y\_2[0] \[Equal] 0\)}
                }], ")"}], ",", "\[IndentingNewLine]", \({y\_1, \ y\_2}\), 
          ",", \({t, 3000}\), ",", 
          " ", \(Method \[Rule] {EventLocator, \ "\<Event\>" \[Rule] 
                y\_2[t], \ "\<Direction\>" \[Rule] \(-1\)}\)}], 
        "]"}]}]], "Input",
  CellLabel->"In[20]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(y\_1\), "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 1614.289572674879`}}, 
                "<>"]\),
              False,
              Editable->False]}], ",", 
          RowBox[{\(y\_2\), "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 1614.289572674879`}}, 
                "<>"]\),
              False,
              Editable->False]}]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[20]="]
}, Open  ]],

Cell["\<\
Note that the event at the initial condition is not \
considered.\
\>", "Text"],

Cell[TextData[{
  "By selecting the endpoint of the ",
  StyleBox["NDSolve", "MR"],
  " solution, it is possible to write a function which returns the period as \
a function of \[Mu]."
}], "Text"],

Cell["\<\
This defines a function which returns the period as a function of \
\[Mu].\
\>", "MathCaption"],

Cell[BoxData[
    RowBox[{
      RowBox[{\(vper[\[Mu]_]\), " ", ":=", " ", 
        RowBox[{"Module", "[", 
          RowBox[{\({vsol}\), ",", "\[IndentingNewLine]", 
            RowBox[{
              RowBox[{"vsol", " ", "=", " ", 
                RowBox[{"First", "[", 
                  RowBox[{\(y\_2\), " ", "/.", " ", 
                    RowBox[{"NDSolve", "[", 
                      RowBox[{
                        RowBox[{"(", GridBox[{
                              {\(\(\(y\_1'\)[
                                    t]\)\(\ \)\(\[Equal]\)\(\ \)\(y\_2[
                                    t]\)\(\ \)\), \(y\_1[0] \[Equal] 2\)},
                              {\(\(y\_2'\)[
                                    t] \[Equal] \[Mu]\ \((1\  - \ 
                                        y\_1[t]^2)\)\ y\_2[t]\  - \ 
                                    y\_1[t]\), \(y\_2[0] \[Equal] 0\)}
                              }], ")"}], ",", 
                        "\[IndentingNewLine]", \({y\_1, \ y\_2}\), 
                        ",", \({t, Max[100, \ 3\ \[Mu]]}\), ",", 
                        " ", \(Method \[Rule] {EventLocator, \ "\<Event\>" \
\[Rule] y\_2[t], \ "\<Direction\>" \[Rule] \(-1\)}\)}], "]"}]}], "]"}]}], ";",
               "\[IndentingNewLine]", \(\(InterpolatingFunctionDomain[
                  vsol]\)[\([1, \(-1\)]\)]\)}]}], "]"}]}], ";"}]], "Input",
  CellLabel->"In[21]:="],

Cell["\<\
This uses the function to compute the period at \[Mu] = 1000\
\>", \
"MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(vper[1000]\)], "Input",
  CellLabel->"In[22]:="],

Cell[BoxData[
    \(1614.289572674879`\)], "Output",
  CellLabel->"Out[22]="]
}, Open  ]],

Cell["\<\
Of course, it is easy to generalize this method to any system with \
periodic solutions.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Poincar\[EAcute] Sections", "Subsection"],

Cell[CellGroupData[{

Cell["The H\[EAcute]non-Heiles system", "Subsubsection"],

Cell["\<\
This is a useful technique for visualizing the solutions of high \
dimensional differential systems.\
\>", "Text"],

Cell["\<\
Define the H\[EAcute]non-Heiles system which models stellar motion \
in a galaxy.\
\>", "Text"],

Cell[TextData[{
  "This gets the ",
  "H\[EAcute]non-Heiles system",
  " from the ",
  StyleBox["NDSolveProblems", "MR"],
  " package."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(system\  = \ 
        GetNDSolveProblem["\<HenonHeiles\>"];\)\), "\[IndentingNewLine]", 
    \(vars\  = \ system["\<DependentVariables\>"[]]; 
    eqns\  = \ {system@"\<System\>"[], \ 
        system@"\<InitialConditions\>"[]}\), "\[IndentingNewLine]", 
    \(\)}], "Input",
  CellLabel->"In[23]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_1\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(Y\_3[T]\)}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_2\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(Y\_4[T]\)}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_3\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(\(-Y\_1[T]\)\ \((1 + 2\ Y\_2[T])\)\)}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_4\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(\(-Y\_1[T]\^2\) + \((\(-1\) + Y\_2[T])\)\ Y\_2[
                    T]\)}]}], "}"}], 
        ",", \({Y\_1[0] \[Equal] 3\/25, Y\_2[0] \[Equal] 3\/25, 
          Y\_3[0] \[Equal] 3\/25, Y\_4[0] \[Equal] 3\/25}\)}], 
      "}"}]], "Output",
  CellLabel->"Out[24]="]
}, Open  ]],

Cell[TextData[{
  "The Poincar\[EAcute]  section of interest in this case is the collection \
of points in the  ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_2 - Y\_4\)]],
  " plane when the orbit passes through ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_1 = \ 0\)]],
  ".  "
}], "Text"],

Cell[TextData[{
  "Since the actual result of the numerical integration is not required, it \
is possible to avoid storing all the data in ",
  StyleBox["InterpolatingFunctions", "MR"],
  " by specifying the output variables list (in the second argument to ",
  StyleBox["NDSolve", "MR"],
  ") as empty, or {}.  This means that ",
  StyleBox["NDSolve", "MR"],
  " will produce no ",
  StyleBox["InterpolatingFunctions", "MR"],
  " as output, avoiding storing a lot of unnecessary data.  ",
  StyleBox["NDSolve", "MR"],
  " does give a message ",
  StyleBox["NDSolve", "MR"],
  "::noout warning there will be no output functions, but it can safely be \
turned off in this case since the data of interest is collected from the \
event actions."
}], "Text"],

Cell["\<\
The linear interpolation event location method is used because the \
purpose of the computation here is to view the results in a graph with \
relatively low resolution.  If you were doing an example where you needed to \
zoom in on the graph to great detail or find a feature, such as a fixed point \
of the Poincar\[EAcute]  map, it would be more appropriate to use the default \
location method.\
\>", "Text"],

Cell["This turns off the message warning about no output.", "MathCaption"],

Cell[BoxData[
    \(\(Off[NDSolve::noout];\)\)], "Input",
  CellLabel->"In[25]:="],

Cell[TextData[{
  "This integrates the H\[EAcute]non-Heiles system using a fourth order \
explicit Runge Kutta method with fixed step size of 0.25.  The event action \
is to ",
  StyleBox["Sow", "MR"],
  " the values of ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_2\ and\ \(\(Y\_4\)\(.\)\)\)]]
}], "MathCaption"],

Cell[BoxData[
    \(\(data = \[IndentingNewLine]Reap[\[IndentingNewLine]\(NDSolve[
              eqns, {}, \ {T, \ 
                10000}, \[IndentingNewLine]Method \[Rule] {EventLocator, \
"\<Event\>" \[Rule] Y\_1[T], "\<EventAction\>" \[RuleDelayed] \ 
                    Sow[{Y\_2[T], \ 
                        Y\_4[T]}], "\<EventLocationMethod\>" -> \
"\<LinearInterpolation\>", \[IndentingNewLine]"\<Method\>" \[Rule] \
{"\<FixedStep\>", "\<Method\>" \[Rule] {"\<ExplicitRungeKutta\>", \
"\<DifferenceOrder\>" \[Rule] 
                            4}}}, \[IndentingNewLine]StartingStepSize \[Rule] 
                0.25, \ 
              MaxSteps \[Rule] \[Infinity]];\)\[IndentingNewLine]];\)\)], \
"Input",
  CellLabel->"In[26]:="],

Cell[TextData[{
  "This plots the  Poincar\[EAcute]  section.  The collected data is found in \
the last part of the result of ",
  StyleBox["Reap", "MR"],
  " and the list of points is the first part of that."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(psdata = data[\([\(-1\), 1]\)];\)\), "\n", 
    \(\(ListPlot[psdata, Axes \[Rule] False, Frame \[Rule] True, 
        AspectRatio \[Rule] 1];\)\)}], "Input",
  CellLabel->"In[31]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.454635 1.9541 0.49927 1.99558 [
[.06382 -0.0125 -12 -9 ]
[.06382 -0.0125 12 0 ]
[.25923 -0.0125 -12 -9 ]
[.25923 -0.0125 12 0 ]
[.45464 -0.0125 -3 -9 ]
[.45464 -0.0125 3 0 ]
[.65005 -0.0125 -9 -9 ]
[.65005 -0.0125 9 0 ]
[.84545 -0.0125 -9 -9 ]
[.84545 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .10015 -24 -4.5 ]
[-0.0125 .10015 0 4.5 ]
[-0.0125 .29971 -24 -4.5 ]
[-0.0125 .29971 0 4.5 ]
[-0.0125 .49927 -6 -4.5 ]
[-0.0125 .49927 0 4.5 ]
[-0.0125 .69883 -18 -4.5 ]
[-0.0125 .69883 0 4.5 ]
[-0.0125 .89839 -18 -4.5 ]
[-0.0125 .89839 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.06382 0 m
.06382 .00625 L
s
[(-0.2)] .06382 -0.0125 0 1 Mshowa
.25923 0 m
.25923 .00625 L
s
[(-0.1)] .25923 -0.0125 0 1 Mshowa
.45464 0 m
.45464 .00625 L
s
[(0)] .45464 -0.0125 0 1 Mshowa
.65005 0 m
.65005 .00625 L
s
[(0.1)] .65005 -0.0125 0 1 Mshowa
.84545 0 m
.84545 .00625 L
s
[(0.2)] .84545 -0.0125 0 1 Mshowa
.125 Mabswid
.1029 0 m
.1029 .00375 L
s
.14198 0 m
.14198 .00375 L
s
.18106 0 m
.18106 .00375 L
s
.22014 0 m
.22014 .00375 L
s
.29831 0 m
.29831 .00375 L
s
.33739 0 m
.33739 .00375 L
s
.37647 0 m
.37647 .00375 L
s
.41555 0 m
.41555 .00375 L
s
.49372 0 m
.49372 .00375 L
s
.5328 0 m
.5328 .00375 L
s
.57188 0 m
.57188 .00375 L
s
.61096 0 m
.61096 .00375 L
s
.68913 0 m
.68913 .00375 L
s
.72821 0 m
.72821 .00375 L
s
.76729 0 m
.76729 .00375 L
s
.80637 0 m
.80637 .00375 L
s
.02473 0 m
.02473 .00375 L
s
.88454 0 m
.88454 .00375 L
s
.92362 0 m
.92362 .00375 L
s
.9627 0 m
.9627 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .10015 m
.00625 .10015 L
s
[(-0.2)] -0.0125 .10015 1 0 Mshowa
0 .29971 m
.00625 .29971 L
s
[(-0.1)] -0.0125 .29971 1 0 Mshowa
0 .49927 m
.00625 .49927 L
s
[(0)] -0.0125 .49927 1 0 Mshowa
0 .69883 m
.00625 .69883 L
s
[(0.1)] -0.0125 .69883 1 0 Mshowa
0 .89839 m
.00625 .89839 L
s
[(0.2)] -0.0125 .89839 1 0 Mshowa
.125 Mabswid
0 .14006 m
.00375 .14006 L
s
0 .17998 m
.00375 .17998 L
s
0 .21989 m
.00375 .21989 L
s
0 .2598 m
.00375 .2598 L
s
0 .33962 m
.00375 .33962 L
s
0 .37953 m
.00375 .37953 L
s
0 .41945 m
.00375 .41945 L
s
0 .45936 m
.00375 .45936 L
s
0 .53918 m
.00375 .53918 L
s
0 .57909 m
.00375 .57909 L
s
0 .619 m
.00375 .619 L
s
0 .65892 m
.00375 .65892 L
s
0 .73874 m
.00375 .73874 L
s
0 .77865 m
.00375 .77865 L
s
0 .81856 m
.00375 .81856 L
s
0 .85848 m
.00375 .85848 L
s
0 .06024 m
.00375 .06024 L
s
0 .02033 m
.00375 .02033 L
s
0 .9383 m
.00375 .9383 L
s
0 .97821 m
.00375 .97821 L
s
.25 Mabswid
0 0 m
0 1 L
s
.06382 .99375 m
.06382 1 L
s
.25923 .99375 m
.25923 1 L
s
.45464 .99375 m
.45464 1 L
s
.65005 .99375 m
.65005 1 L
s
.84545 .99375 m
.84545 1 L
s
.125 Mabswid
.1029 .99625 m
.1029 1 L
s
.14198 .99625 m
.14198 1 L
s
.18106 .99625 m
.18106 1 L
s
.22014 .99625 m
.22014 1 L
s
.29831 .99625 m
.29831 1 L
s
.33739 .99625 m
.33739 1 L
s
.37647 .99625 m
.37647 1 L
s
.41555 .99625 m
.41555 1 L
s
.49372 .99625 m
.49372 1 L
s
.5328 .99625 m
.5328 1 L
s
.57188 .99625 m
.57188 1 L
s
.61096 .99625 m
.61096 1 L
s
.68913 .99625 m
.68913 1 L
s
.72821 .99625 m
.72821 1 L
s
.76729 .99625 m
.76729 1 L
s
.80637 .99625 m
.80637 1 L
s
.02473 .99625 m
.02473 1 L
s
.88454 .99625 m
.88454 1 L
s
.92362 .99625 m
.92362 1 L
s
.9627 .99625 m
.9627 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .10015 m
1 .10015 L
s
.99375 .29971 m
1 .29971 L
s
.99375 .49927 m
1 .49927 L
s
.99375 .69883 m
1 .69883 L
s
.99375 .89839 m
1 .89839 L
s
.125 Mabswid
.99625 .14006 m
1 .14006 L
s
.99625 .17998 m
1 .17998 L
s
.99625 .21989 m
1 .21989 L
s
.99625 .2598 m
1 .2598 L
s
.99625 .33962 m
1 .33962 L
s
.99625 .37953 m
1 .37953 L
s
.99625 .41945 m
1 .41945 L
s
.99625 .45936 m
1 .45936 L
s
.99625 .53918 m
1 .53918 L
s
.99625 .57909 m
1 .57909 L
s
.99625 .619 m
1 .619 L
s
.99625 .65892 m
1 .65892 L
s
.99625 .73874 m
1 .73874 L
s
.99625 .77865 m
1 .77865 L
s
.99625 .81856 m
1 .81856 L
s
.99625 .85848 m
1 .85848 L
s
.99625 .06024 m
1 .06024 L
s
.99625 .02033 m
1 .02033 L
s
.99625 .9383 m
1 .9383 L
s
.99625 .97821 m
1 .97821 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
.008 w
.47982 .16262 Mdot
.5749 .79458 Mdot
.49091 .15422 Mdot
.5826 .80392 Mdot
.56633 .78426 Mdot
.45621 .18323 Mdot
.46819 .17224 Mdot
.60188 .82655 Mdot
.51039 .14037 Mdot
.51892 .13461 Mdot
.50117 .14687 Mdot
.58943 .81239 Mdot
.59582 .8199 Mdot
.4082 .24187 Mdot
.51245 .71787 Mdot
.52438 .73287 Mdot
.50022 .70234 Mdot
.38866 .27839 Mdot
.39787 .25973 Mdot
.44384 .19569 Mdot
.54697 .76045 Mdot
.55711 .77285 Mdot
.53589 .74713 Mdot
.41943 .22511 Mdot
.43144 .20966 Mdot
.65907 .85815 Mdot
.55471 .0978 Mdot
.55433 .09275 Mdot
.5535 .10248 Mdot
.64298 .8552 Mdot
.65043 .85707 Mdot
.69152 .85668 Mdot
.54929 .08168 Mdot
.54313 .07544 Mdot
.55268 .08741 Mdot
.66846 .85866 Mdot
.6791 .8583 Mdot
.61844 .84237 Mdot
.53338 .1246 Mdot
.53908 .12004 Mdot
.52667 .12942 Mdot
.60738 .83255 Mdot
.61279 .83783 Mdot
.6362 .85279 Mdot
.54833 .11138 Mdot
.55128 .10697 Mdot
.54416 .11569 Mdot
.62394 .84644 Mdot
.62972 .84995 Mdot
.40763 .4839 Mdot
.35887 .52098 Mdot
.40699 .49393 Mdot
.35895 .51023 Mdot
.35875 .53197 Mdot
.41084 .46316 Mdot
.40891 .47366 Mdot
.35883 .47836 Mdot
.40755 .51397 Mdot
.40877 .52417 Mdot
.40696 .50391 Mdot
.35897 .49962 Mdot
.35893 .48906 Mdot
.42613 .41627 Mdot
.35978 .59644 Mdot
.35904 .5819 Mdot
.36106 .61195 Mdot
.43921 .38899 Mdot
.4322 .40296 Mdot
.41346 .45227 Mdot
.35861 .55555 Mdot
.35864 .54346 Mdot
.35871 .56833 Mdot
.42106 .42888 Mdot
.41687 .44085 Mdot
.36548 .35355 Mdot
.44679 .62348 Mdot
.45582 .6387 Mdot
.43871 .60883 Mdot
.36076 .38745 Mdot
.36266 .37098 Mdot
.38079 .29746 Mdot
.47678 .67035 Mdot
.48828 .68641 Mdot
.46582 .65439 Mdot
.36935 .33533 Mdot
.37438 .31656 Mdot
.3585 .44381 Mdot
.41324 .54557 Mdot
.41656 .55695 Mdot
.41064 .53466 Mdot
.35871 .46732 Mdot
.35859 .45583 Mdot
.3596 .40291 Mdot
.42577 .58155 Mdot
.43174 .59484 Mdot
.4207 .56892 Mdot
.35857 .43106 Mdot
.3589 .41744 Mdot
.50007 .8515 Mdot
.59555 .17902 Mdot
.48957 .84399 Mdot
.58907 .18671 Mdot
.6017 .17225 Mdot
.51821 .86397 Mdot
.50951 .85812 Mdot
.5654 .21547 Mdot
.4664 .82551 Mdot
.4542 .81421 Mdot
.47826 .83539 Mdot
.58208 .19536 Mdot
.57418 .20491 Mdot
.54401 .88299 Mdot
.62959 .14846 Mdot
.62387 .15204 Mdot
.63601 .14557 Mdot
.55139 .89166 Mdot
.54831 .88727 Mdot
.52613 .86922 Mdot
.61269 .16076 Mdot
.60727 .16615 Mdot
.61835 .15615 Mdot
.53884 .87864 Mdot
.53302 .87406 Mdot
.46327 .34757 Mdot
.37249 .67811 Mdot
.36768 .65922 Mdot
.37871 .69741 Mdot
.48578 .31523 Mdot
.47422 .33147 Mdot
.43646 .3933 Mdot
.3614 .6235 Mdot
.35966 .60706 Mdot
.36402 .64094 Mdot
.45333 .36335 Mdot
.4444 .37863 Mdot
.53417 .25344 Mdot
.4291 .78703 Mdot
.417 .77118 Mdot
.44164 .8014 Mdot
.55593 .22715 Mdot
.54551 .23982 Mdot
.49784 .29907 Mdot
.39549 .73576 Mdot
.38641 .71676 Mdot
.40576 .75401 Mdot
.5224 .268 Mdot
.51024 .28328 Mdot
.93311 .60717 Mdot
.15406 .16944 Mdot
.94897 .49515 Mdot
.04722 .37218 Mdot
.27693 .07928 Mdot
.85438 .75216 Mdot
.89533 .69413 Mdot
.28463 .92282 Mdot
.89244 .29935 Mdot
.8517 .24302 Mdot
.9309 .38403 Mdot
.0521 .6446 Mdot
.16326 .83886 Mdot
.74007 .84168 Mdot
.52073 .06191 Mdot
.50124 .05496 Mdot
.53419 .06886 Mdot
.70532 .85388 Mdot
.7215 .84902 Mdot
.8182 .78957 Mdot
.43125 .04544 Mdot
.36895 .05053 Mdot
.47283 .04878 Mdot
.76179 .83068 Mdot
.78749 .81428 Mdot
.55315 .91107 Mdot
.67852 .13987 Mdot
.66803 .13959 Mdot
.69083 .14144 Mdot
.54396 .92292 Mdot
.54991 .91674 Mdot
.55365 .89613 Mdot
.65014 .14123 Mdot
.64279 .14315 Mdot
.65868 .14011 Mdot
.55475 .90576 Mdot
.55497 .90076 Mdot
.4755 .94958 Mdot
.78572 .18276 Mdot
.76025 .16671 Mdot
.81594 .20671 Mdot
.3743 .94904 Mdot
.43491 .95322 Mdot
.5352 .92947 Mdot
.72041 .14883 Mdot
.70451 .14417 Mdot
.73888 .15606 Mdot
.50307 .94332 Mdot
.52219 .93636 Mdot
.57774 .79816 Mdot
.47378 .16887 Mdot
.58489 .8069 Mdot
.48486 .16001 Mdot
.46231 .17899 Mdot
.56115 .77783 Mdot
.56973 .78853 Mdot
.51359 .13936 Mdot
.59738 .82179 Mdot
.60311 .828 Mdot
.59128 .8148 Mdot
.49535 .15226 Mdot
.50497 .14544 Mdot
.51887 .72569 Mdot
.40421 .25048 Mdot
.41488 .23354 Mdot
.3945 .26838 Mdot
.49518 .69527 Mdot
.50696 .71074 Mdot
.55169 .76616 Mdot
.43816 .20341 Mdot
.45033 .19048 Mdot
.42622 .2178 Mdot
.53032 .74004 Mdot
.54125 .75359 Mdot
.55362 .09785 Mdot
.65167 .85623 Mdot
.66046 .85709 Mdot
.644 .85462 Mdot
.55105 .10704 Mdot
.55292 .10257 Mdot
.54658 .08165 Mdot
.681 .85668 Mdot
.69354 .85481 Mdot
.66998 .8574 Mdot
.55276 .09278 Mdot
.55071 .08743 Mdot
.53489 .12436 Mdot
.61376 .83851 Mdot
.61931 .84275 Mdot
.60841 .83361 Mdot
.52159 .13394 Mdot
.52878 .12901 Mdot
.5485 .11142 Mdot
.63063 .84979 Mdot
.63717 .8524 Mdot
.62476 .84656 Mdot
.54019 .11995 Mdot
.54485 .11569 Mdot
.35979 .51214 Mdot
.40838 .48214 Mdot
.35982 .50173 Mdot
.40763 .49206 Mdot
.40972 .47205 Mdot
.35959 .53365 Mdot
.3597 .52271 Mdot
.40899 .52175 Mdot
.35972 .48079 Mdot
.35959 .47007 Mdot
.3598 .49131 Mdot
.40749 .50188 Mdot
.40793 .51173 Mdot
.36 .5831 Mdot
.42712 .41524 Mdot
.42202 .42773 Mdot
.43308 .40212 Mdot
.362 .61271 Mdot
.36076 .59746 Mdot
.3595 .54504 Mdot
.41441 .45085 Mdot
.41171 .46166 Mdot
.41782 .43958 Mdot
.35959 .56961 Mdot
.35946 .55695 Mdot
.45276 .6329 Mdot
.36522 .3602 Mdot
.36864 .34256 Mdot
.36283 .37697 Mdot
.4367 .60392 Mdot
.44419 .61811 Mdot
.48367 .67954 Mdot
.37895 .30563 Mdot
.38602 .28689 Mdot
.3732 .32428 Mdot
.46221 .64817 Mdot
.47252 .66378 Mdot
.4161 .55369 Mdot
.35939 .44719 Mdot
.35941 .43485 Mdot
.35946 .4589 Mdot
.41068 .532 Mdot
.41301 .5426 Mdot
.43019 .59039 Mdot
.3602 .40776 Mdot
.36123 .39282 Mdot
.35963 .42176 Mdot
.41993 .56529 Mdot
.42458 .5775 Mdot
.50997 .8571 Mdot
.60088 .1728 Mdot
.50061 .8507 Mdot
.59505 .17944 Mdot
.60658 .16697 Mdot
.52575 .86799 Mdot
.51824 .86283 Mdot
.57431 .20448 Mdot
.47963 .8351 Mdot
.46822 .82557 Mdot
.49041 .84342 Mdot
.58889 .18695 Mdot
.58194 .19524 Mdot
.54726 .88577 Mdot
.63454 .14674 Mdot
.6285 .14971 Mdot
.64143 .14446 Mdot
.5525 .89448 Mdot
.55051 .89003 Mdot
.53254 .87272 Mdot
.61715 .1571 Mdot
.61186 .16172 Mdot
.62284 .15319 Mdot
.54311 .88154 Mdot
.53835 .87719 Mdot
.47715 .32768 Mdot
.38136 .7016 Mdot
.3749 .68262 Mdot
.38911 .72049 Mdot
.50058 .29593 Mdot
.48873 .31174 Mdot
.44723 .37438 Mdot
.36579 .64573 Mdot
.363 .6284 Mdot
.36972 .66389 Mdot
.46634 .34358 Mdot
.45635 .3592 Mdot
.54668 .23827 Mdot
.44388 .80238 Mdot
.43171 .78856 Mdot
.45618 .8147 Mdot
.56606 .21478 Mdot
.55685 .22604 Mdot
.51255 .28048 Mdot
.40864 .75669 Mdot
.39827 .73898 Mdot
.41992 .77329 Mdot
.53594 .25153 Mdot
.52453 .26565 Mdot
.13152 .19936 Mdot
.90086 .6792 Mdot
.93565 .5881 Mdot
.86014 .7415 Mdot
.35237 .05601 Mdot
.25498 .09277 Mdot
.18444 .85609 Mdot
.92293 .36857 Mdot
.88377 .28992 Mdot
.94562 .47513 Mdot
.03967 .41965 Mdot
.0675 .68442 Mdot
.51522 .06202 Mdot
.72426 .84619 Mdot
.74308 .83831 Mdot
.70763 .85162 Mdot
.53978 .07543 Mdot
.53 .06888 Mdot
.41895 .04772 Mdot
.79163 .80874 Mdot
.82312 .78207 Mdot
.76552 .82624 Mdot
.49439 .05534 Mdot
.46367 .04973 Mdot
.54883 .91473 Mdot
.68823 .14245 Mdot
.67647 .14114 Mdot
.70185 .1452 Mdot
.53471 .92717 Mdot
.5434 .92072 Mdot
.55369 .89907 Mdot
.65679 .14132 Mdot
.6487 .14259 Mdot
.66621 .14091 Mdot
.55202 .90917 Mdot
.55375 .90391 Mdot
.43992 .95088 Mdot
.80934 .20375 Mdot
.78044 .18158 Mdot
.84403 .23773 Mdot
.29978 .92671 Mdot
.38341 .94815 Mdot
.52244 .93387 Mdot
.73517 .15642 Mdot
.7171 .14948 Mdot
.75578 .16641 Mdot
.47825 .94691 Mdot
.50416 .9407 Mdot
.60889 .83403 Mdot
.51895 .13244 Mdot
.61474 .83951 Mdot
.52697 .12697 Mdot
.50976 .13848 Mdot
.59685 .82058 Mdot
.60298 .82775 Mdot
.54584 .11283 Mdot
.626 .84867 Mdot
.63225 .8522 Mdot
.62029 .84442 Mdot
.5341 .12197 Mdot
.54047 .11732 Mdot
.56524 .78256 Mdot
.45183 .18442 Mdot
.46472 .17255 Mdot
.43885 .19789 Mdot
.54409 .75723 Mdot
.55503 .77046 Mdot
.59005 .81259 Mdot
.48877 .15317 Mdot
.49963 .14532 Mdot
.47716 .16219 Mdot
.57432 .79364 Mdot
.58247 .80367 Mdot
.53497 .06427 Mdot
.71085 .85573 Mdot
.728 .85028 Mdot
.69623 .85892 Mdot
.55107 .07765 Mdot
.54467 .07117 Mdot
.46687 .04358 Mdot
.77095 .82974 Mdot
.79825 .81144 Mdot
.74755 .84223 Mdot
.52007 .05696 Mdot
.49877 .04976 Mdot
.55586 .09928 Mdot
.64562 .85779 Mdot
.65364 .85954 Mdot
.63867 .85527 Mdot
.54999 .10835 Mdot
.5534 .10388 Mdot
.55501 .08368 Mdot
.67209 .86123 Mdot
.68352 .86058 Mdot
.66235 .86074 Mdot
.55686 .09438 Mdot
.55664 .0892 Mdot
.35711 .47638 Mdot
.40658 .51563 Mdot
.35697 .4649 Mdot
.40797 .5262 Mdot
.40587 .50523 Mdot
.35728 .4984 Mdot
.35723 .48746 Mdot
.41661 .56038 Mdot
.35679 .4403 Mdot
.35689 .42691 Mdot
.35685 .45291 Mdot
.41005 .5371 Mdot
.41293 .54848 Mdot
.35693 .54364 Mdot
.41244 .45196 Mdot
.40972 .4632 Mdot
.41601 .44015 Mdot
.35699 .5694 Mdot
.35689 .55615 Mdot
.35726 .50935 Mdot
.40644 .48459 Mdot
.40583 .49493 Mdot
.40773 .47403 Mdot
.35704 .53177 Mdot
.35718 .52043 Mdot
.49473 .69578 Mdot
.38311 .28617 Mdot
.39199 .26655 Mdot
.37575 .30601 Mdot
.47057 .66261 Mdot
.48236 .6792 Mdot
.53242 .74299 Mdot
.41383 .22959 Mdot
.42614 .21297 Mdot
.40232 .24755 Mdot
.50732 .71211 Mdot
.52 .72792 Mdot
.43333 .60016 Mdot
.35818 .39718 Mdot
.35965 .38078 Mdot
.35731 .41256 Mdot
.42116 .57292 Mdot
.42674 .58618 Mdot
.45973 .64623 Mdot
.36522 .34494 Mdot
.3698 .32573 Mdot
.36189 .36336 Mdot
.44098 .61487 Mdot
.44983 .63026 Mdot
.54647 .23841 Mdot
.44108 .80399 Mdot
.53477 .25262 Mdot
.42813 .78906 Mdot
.45422 .81727 Mdot
.56754 .21326 Mdot
.55739 .22525 Mdot
.39299 .73549 Mdot
.50942 .28358 Mdot
.49663 .30002 Mdot
.52227 .26771 Mdot
.41552 .77252 Mdot
.40367 .75457 Mdot
.5988 .17581 Mdot
.51209 .86218 Mdot
.50207 .85551 Mdot
.52116 .86807 Mdot
.61073 .16266 Mdot
.60487 .1688 Mdot
.57653 .2023 Mdot
.47969 .83906 Mdot
.46721 .82892 Mdot
.49131 .84786 Mdot
.59207 .18364 Mdot
.58459 .19241 Mdot
.35615 .5726 Mdot
.42108 .42468 Mdot
.41638 .43742 Mdot
.42684 .41119 Mdot
.35756 .60288 Mdot
.35662 .58722 Mdot
.35617 .53398 Mdot
.40962 .46106 Mdot
.40747 .47213 Mdot
.41259 .44951 Mdot
.35602 .55897 Mdot
.35606 .54616 Mdot
.36983 .67563 Mdot
.47194 .33355 Mdot
.4608 .35016 Mdot
.48402 .31676 Mdot
.38375 .71571 Mdot
.37607 .69563 Mdot
.35911 .61959 Mdot
.44151 .38202 Mdot
.43362 .39698 Mdot
.45063 .36637 Mdot
.36501 .6561 Mdot
.3615 .63734 Mdot
.40308 .95669 Mdot
.84143 .2252 Mdot
.45596 .95637 Mdot
.80684 .19353 Mdot
.88177 .27429 Mdot
.2089 .88398 Mdot
.32341 .94173 Mdot
.73332 .14997 Mdot
.51521 .94382 Mdot
.53176 .93649 Mdot
.49149 .95086 Mdot
.77818 .17299 Mdot
.75372 .15902 Mdot
.24356 .09187 Mdot
.87018 .73981 Mdot
.9125 .67248 Mdot
.83127 .78326 Mdot
.41973 .0412 Mdot
.34859 .05021 Mdot
.08038 .72835 Mdot
.95163 .4558 Mdot
.92351 .34975 Mdot
.94656 .57434 Mdot
.11303 .21449 Mdot
.02947 .4597 Mdot
.63427 .1451 Mdot
.55128 .89182 Mdot
.54734 .88735 Mdot
.55441 .89631 Mdot
.64797 .13984 Mdot
.64085 .14219 Mdot
.61659 .15734 Mdot
.53598 .87833 Mdot
.52907 .87341 Mdot
.54215 .88291 Mdot
.62793 .14848 Mdot
.62218 .15259 Mdot
.68709 .13801 Mdot
.54993 .9229 Mdot
.55449 .91677 Mdot
.54263 .9295 Mdot
.71546 .1438 Mdot
.70029 .14008 Mdot
.6562 .13826 Mdot
.5573 .90588 Mdot
.55658 .90094 Mdot
.55662 .91115 Mdot
.67527 .13705 Mdot
.66522 .13729 Mdot
.49937 .14684 Mdot
.58895 .81167 Mdot
.50906 .14007 Mdot
.59572 .81947 Mdot
.58178 .80284 Mdot
.47726 .16337 Mdot
.48878 .15454 Mdot
.61315 .83816 Mdot
.52612 .12874 Mdot
.53305 .12378 Mdot
.51804 .1341 Mdot
.60186 .82645 Mdot
.60751 .83271 Mdot
.42752 .21288 Mdot
.53294 .74361 Mdot
.54446 .75746 Mdot
.52092 .72885 Mdot
.40406 .24655 Mdot
.41536 .22906 Mdot
.46514 .17349 Mdot
.56477 .78228 Mdot
.57381 .79304 Mdot
.55503 .77039 Mdot
.44012 .19819 Mdot
.4527 .18507 Mdot
.68 .85959 Mdot
.55407 .08612 Mdot
.55075 .08032 Mdot
.55555 .09152 Mdot
.6598 .85934 Mdot
.66923 .85993 Mdot
.72288 .85032 Mdot
.53575 .06729 Mdot
.52221 .06019 Mdot
.54466 .07398 Mdot
.69252 .85801 Mdot
.70646 .85522 Mdot
.63026 .85077 Mdot
.5445 .11472 Mdot
.54884 .11034 Mdot
.53914 .11913 Mdot
.61883 .84291 Mdot
.62437 .84715 Mdot
.6511 .85819 Mdot
.55446 .10136 Mdot
.5558 .09662 Mdot
.55201 .10588 Mdot
.63679 .85373 Mdot
.64356 .85626 Mdot
.40633 .49839 Mdot
.35812 .50558 Mdot
.40659 .50854 Mdot
.35811 .49488 Mdot
.35805 .5164 Mdot
.40774 .47793 Mdot
.40672 .48821 Mdot
.35778 .46162 Mdot
.40905 .52931 Mdot
.41132 .54016 Mdot
.40749 .51879 Mdot
.35805 .48405 Mdot
.35793 .473 Mdot
.41894 .43315 Mdot
.35804 .57722 Mdot
.35777 .56367 Mdot
.35862 .59167 Mdot
.42964 .40728 Mdot
.42384 .42054 Mdot
.40942 .46739 Mdot
.35784 .53898 Mdot
.35795 .52749 Mdot
.35774 .55096 Mdot
.41498 .44509 Mdot
.41184 .45646 Mdot
.37144 .32316 Mdot
.46169 .64863 Mdot
.47252 .66479 Mdot
.45178 .63285 Mdot
.36337 .3603 Mdot
.36679 .34209 Mdot
.39394 .26505 Mdot
.49622 .69739 Mdot
.50865 .71336 Mdot
.48405 .68112 Mdot
.37752 .30376 Mdot
.38505 .28425 Mdot
.35786 .42364 Mdot
.41817 .56335 Mdot
.42289 .57586 Mdot
.41432 .55147 Mdot
.35766 .4497 Mdot
.35766 .43707 Mdot
.36091 .37766 Mdot
.43524 .60298 Mdot
.443 .6176 Mdot
.42852 .58906 Mdot
.35833 .40934 Mdot
.35928 .39403 Mdot
.58025 .19764 Mdot
.48601 .84259 Mdot
.57198 .2078 Mdot
.47418 .83339 Mdot
.49691 .85061 Mdot
.59465 .18043 Mdot
.58768 .1885 Mdot
.4363 .79704 Mdot
.55247 .23122 Mdot
.54153 .24457 Mdot
.5626 .21894 Mdot
.46175 .82282 Mdot
.44905 .81072 Mdot
.61807 .15619 Mdot
.53815 .87899 Mdot
.53181 .8743 Mdot
.54375 .88342 Mdot
.6294 .14806 Mdot
.62358 .15181 Mdot
.60094 .17319 Mdot
.51625 .86376 Mdot
.50695 .85761 Mdot
.52461 .86925 Mdot
.61237 .16109 Mdot
.60669 .16673 Mdot
.36167 .63202 Mdot
.44821 .37111 Mdot
.43967 .38632 Mdot
.45792 .35531 Mdot
.36904 .66911 Mdot
.36474 .65015 Mdot
.35733 .58349 Mdot
.42577 .41468 Mdot
.42043 .42776 Mdot
.43221 .40085 Mdot
.35952 .61482 Mdot
.35812 .59863 Mdot
.40033 .74694 Mdot
.51733 .27398 Mdot
.50483 .28982 Mdot
.52968 .25884 Mdot
.42364 .78176 Mdot
.41151 .76498 Mdot
.37473 .68865 Mdot
.48006 .32258 Mdot
.46859 .33907 Mdot
.49229 .30609 Mdot
.39043 .72796 Mdot
.38188 .7084 Mdot
.93111 .37665 Mdot
.05547 .66212 Mdot
.17226 .85059 Mdot
.04201 .38615 Mdot
.9372 .60217 Mdot
.95141 .488 Mdot
.81481 .20291 Mdot
.38177 .95231 Mdot
.44092 .9552 Mdot
.29364 .92893 Mdot
.89151 .29327 Mdot
.8505 .23824 Mdot
.78984 .81508 Mdot
.47371 .04672 Mdot
.4313 .04328 Mdot
.50255 .05308 Mdot
.74167 .84293 Mdot
.76374 .83175 Mdot
.8995 .69168 Mdot
.27281 .07884 Mdot
.14707 .1741 Mdot
.36745 .04859 Mdot
.8211 .78983 Mdot
.85794 .75144 Mdot
.65836 .13905 Mdot
.55599 .90637 Mdot
.55597 .90135 Mdot
.5548 .91168 Mdot
.67805 .13849 Mdot
.66758 .13834 Mdot
.63583 .145 Mdot
.55171 .89222 Mdot
.54833 .88779 Mdot
.55437 .89669 Mdot
.64985 .14031 Mdot
.64249 .14236 Mdot
.73806 .15376 Mdot
.5071 .94435 Mdot
.52551 .93721 Mdot
.48038 .95093 Mdot
.78469 .17962 Mdot
.75937 .16407 Mdot
.69027 .13987 Mdot
.54631 .9236 Mdot
.55189 .91737 Mdot
.53804 .9302 Mdot
.71975 .14687 Mdot
.70384 .14239 Mdot
.61471 .83844 Mdot
.53167 .12967 Mdot
.61983 .84214 Mdot
.53683 .12545 Mdot
.52602 .13416 Mdot
.60445 .82967 Mdot
.60963 .83429 Mdot
.54731 .11308 Mdot
.63143 .84791 Mdot
.63774 .85018 Mdot
.62552 .84522 Mdot
.54122 .12134 Mdot
.54455 .11721 Mdot
.58249 .8045 Mdot
.48544 .16357 Mdot
.4951 .15631 Mdot
.47522 .1718 Mdot
.56859 .78742 Mdot
.57594 .79635 Mdot
.59946 .82434 Mdot
.5119 .1441 Mdot
.51938 .13892 Mdot
.50387 .14986 Mdot
.58842 .81189 Mdot
.59417 .81843 Mdot
.50392 .06499 Mdot
.7265 .83964 Mdot
.74593 .83055 Mdot
.70992 .84566 Mdot
.53169 .07767 Mdot
.52024 .07137 Mdot
.39843 .05554 Mdot
.79516 .79799 Mdot
.82699 .76887 Mdot
.76842 .81743 Mdot
.48056 .05906 Mdot
.44698 .05485 Mdot
.54971 .09969 Mdot
.65282 .85296 Mdot
.66152 .85352 Mdot
.64503 .85178 Mdot
.54925 .10885 Mdot
.54989 .10436 Mdot
.53972 .08373 Mdot
.68263 .85206 Mdot
.69545 .8496 Mdot
.67162 .85316 Mdot
.54823 .09473 Mdot
.5448 .08937 Mdot
.36224 .42407 Mdot
.42094 .56359 Mdot
.36272 .41094 Mdot
.42525 .57515 Mdot
.41741 .5526 Mdot
.36196 .44815 Mdot
.362 .43647 Mdot
.44307 .61334 Mdot
.365 .38216 Mdot
.3671 .36648 Mdot
.36357 .397 Mdot
.4303 .58727 Mdot
.43624 .60001 Mdot
.36234 .49008 Mdot
.40922 .50326 Mdot
.40972 .51268 Mdot
.40927 .49392 Mdot
.36235 .50991 Mdot
.36237 .49998 Mdot
.36205 .4592 Mdot
.41235 .53197 Mdot
.41457 .5421 Mdot
.41075 .52221 Mdot
.36226 .48011 Mdot
.36215 .46983 Mdot
.53275 .74257 Mdot
.42043 .23103 Mdot
.43116 .21665 Mdot
.41015 .24658 Mdot
.51125 .71538 Mdot
.52225 .72929 Mdot
.56078 .77751 Mdot
.45361 .19168 Mdot
.46468 .18114 Mdot
.44231 .20351 Mdot
.54268 .75511 Mdot
.55217 .76672 Mdot
.46872 .65628 Mdot
.37381 .33305 Mdot
.37881 .31553 Mdot
.36995 .35009 Mdot
.45073 .62722 Mdot
.45933 .64158 Mdot
.5003 .70091 Mdot
.39219 .28027 Mdot
.40063 .26308 Mdot
.38496 .29783 Mdot
.47872 .6712 Mdot
.4894 .68613 Mdot
.55335 .23028 Mdot
.45458 .8087 Mdot
.54398 .24199 Mdot
.44326 .79689 Mdot
.46565 .81921 Mdot
.57002 .20965 Mdot
.56195 .21946 Mdot
.41038 .75363 Mdot
.52315 .26791 Mdot
.51216 .28197 Mdot
.53375 .25452 Mdot
.43184 .78373 Mdot
.42076 .76926 Mdot
.5954 .17874 Mdot
.51333 .85587 Mdot
.50511 .85021 Mdot
.52067 .861 Mdot
.60583 .1677 Mdot
.60058 .17285 Mdot
.57718 .20067 Mdot
.48683 .83664 Mdot
.4765 .82848 Mdot
.49629 .84384 Mdot
.58983 .18529 Mdot
.58365 .19252 Mdot
.36285 .60137 Mdot
.43577 .39868 Mdot
.42975 .41157 Mdot
.44265 .38519 Mdot
.36633 .63221 Mdot
.36427 .61638 Mdot
.36125 .56145 Mdot
.42029 .43551 Mdot
.41671 .44664 Mdot
.42459 .42385 Mdot
.36193 .58721 Mdot
.36144 .57392 Mdot
.38455 .70179 Mdot
.48982 .3115 Mdot
.47906 .32662 Mdot
.50087 .29653 Mdot
.40076 .737 Mdot
.39207 .71961 Mdot
.36931 .64886 Mdot
.45924 .3566 Mdot
.45052 .37113 Mdot
.46873 .3417 Mdot
.37835 .6839 Mdot
.3733 .66618 Mdot
.43941 .94428 Mdot
.80084 .20499 Mdot
.47545 .94067 Mdot
.77343 .18419 Mdot
.83355 .23634 Mdot
.31023 .92357 Mdot
.38689 .94192 Mdot
.71297 .15353 Mdot
.51797 .92861 Mdot
.53008 .92231 Mdot
.50023 .93497 Mdot
.74994 .16981 Mdot
.73029 .16026 Mdot
.10819 .24441 Mdot
.90292 .65832 Mdot
.9327 .56444 Mdot
.86387 .72491 Mdot
.32701 .06952 Mdot
.22596 .11823 Mdot
.20406 .86422 Mdot
.90975 .35562 Mdot
.8713 .28404 Mdot
.93579 .45439 Mdot
.04001 .47458 Mdot
.08856 .71798 Mdot
.62689 .1524 Mdot
.54545 .88252 Mdot
.54199 .87846 Mdot
.54814 .88663 Mdot
.63949 .14772 Mdot
.63276 .14974 Mdot
.61088 .16309 Mdot
.5329 .87013 Mdot
.52704 .86574 Mdot
.53798 .87431 Mdot
.6213 .15547 Mdot
.61586 .15896 Mdot
.67372 .14514 Mdot
.54443 .91049 Mdot
.54787 .90516 Mdot
.53874 .91622 Mdot
.69827 .1493 Mdot
.68516 .14656 Mdot
.64666 .14611 Mdot
.55032 .8954 Mdot
.5496 .89095 Mdot
.54995 .90009 Mdot
.66375 .14479 Mdot
.65456 .14503 Mdot
.5485 .10803 Mdot
.6404 .84977 Mdot
.5488 .10355 Mdot
.64772 .85118 Mdot
.63354 .84796 Mdot
.54522 .11648 Mdot
.54748 .11236 Mdot
.67542 .85146 Mdot
.54534 .0937 Mdot
.54135 .08834 Mdot
.54794 .09881 Mdot
.65582 .85204 Mdot
.66523 .85206 Mdot
.52881 .13291 Mdot
.60653 .83142 Mdot
.61154 .83565 Mdot
.60149 .82666 Mdot
.51629 .14225 Mdot
.52304 .13743 Mdot
.54222 .12052 Mdot
.62191 .84271 Mdot
.62758 .84552 Mdot
.61647 .83948 Mdot
.53409 .12868 Mdot
.53868 .1246 Mdot
.8774 .70041 Mdot
.28883 .08595 Mdot
.17894 .15915 Mdot
.37072 .06143 Mdot
.80601 .78589 Mdot
.83939 .75202 Mdot
.92603 .41464 Mdot
.04684 .57435 Mdot
.13036 .78385 Mdot
.07163 .32577 Mdot
.91427 .62379 Mdot
.93517 .52193 Mdot
.71565 .8419 Mdot
.52543 .07652 Mdot
.51212 .07029 Mdot
.53484 .08257 Mdot
.68721 .84969 Mdot
.70054 .84663 Mdot
.77764 .80863 Mdot
.46604 .05895 Mdot
.42731 .05647 Mdot
.493 .06417 Mdot
.73336 .83458 Mdot
.75357 .82416 Mdot
.45854 .63965 Mdot
.37052 .35225 Mdot
.46768 .65405 Mdot
.37415 .33562 Mdot
.36776 .3683 Mdot
.44284 .612 Mdot
.45027 .62559 Mdot
.39173 .28389 Mdot
.48766 .68336 Mdot
.49835 .6979 Mdot
.47738 .66868 Mdot
.3789 .3185 Mdot
.38478 .30115 Mdot
.42135 .56321 Mdot
.36286 .43682 Mdot
.36309 .42466 Mdot
.36285 .44825 Mdot
.41512 .54207 Mdot
.41792 .55241 Mdot
.4362 .59893 Mdot
.36439 .39812 Mdot
.36574 .38363 Mdot
.36358 .41176 Mdot
.42554 .57456 Mdot
.43048 .58645 Mdot
.48233 .16736 Mdot
.573 .79289 Mdot
.57963 .80109 Mdot
.5657 .78388 Mdot
.46187 .1852 Mdot
.47226 .17573 Mdot
.50877 .14752 Mdot
.59124 .81523 Mdot
.59657 .82122 Mdot
.58555 .80857 Mdot
.49194 .15996 Mdot
.50075 .1534 Mdot
.41879 .23524 Mdot
.51973 .72593 Mdot
.53014 .73908 Mdot
.50903 .71217 Mdot
.39974 .26698 Mdot
.40885 .25067 Mdot
.45099 .19585 Mdot
.54928 .76316 Mdot
.55792 .77392 Mdot
.53992 .75156 Mdot
.42919 .22089 Mdot
.43996 .20775 Mdot
.51224 .28205 Mdot
.41119 .75305 Mdot
.50138 .29636 Mdot
.40172 .73678 Mdot
.4215 .76836 Mdot
.53347 .25521 Mdot
.52297 .26827 Mdot
.37962 .68478 Mdot
.47991 .32585 Mdot
.46988 .34069 Mdot
.49043 .311 Mdot
.39332 .71981 Mdot
.3859 .70237 Mdot
.56868 .21106 Mdot
.47583 .82651 Mdot
.46533 .81735 Mdot
.48572 .83459 Mdot
.58232 .19427 Mdot
.57571 .20221 Mdot
.54323 .24285 Mdot
.44322 .79541 Mdot
.43229 .78252 Mdot
.45434 .80701 Mdot
.56102 .22081 Mdot
.55241 .23136 Mdot
.36207 .55202 Mdot
.41793 .44445 Mdot
.41497 .45501 Mdot
.42154 .43342 Mdot
.36239 .57616 Mdot
.36212 .56372 Mdot
.36228 .51993 Mdot
.41098 .47494 Mdot
.40984 .48452 Mdot
.41267 .46515 Mdot
.36211 .54087 Mdot
.36218 .53018 Mdot
.36758 .63395 Mdot
.45182 .36966 Mdot
.44405 .38354 Mdot
.4604 .35534 Mdot
.37455 .66736 Mdot
.37062 .65038 Mdot
.36297 .58939 Mdot
.43113 .40965 Mdot
.42594 .42182 Mdot
.43711 .39689 Mdot
.36537 .61824 Mdot
.3639 .6034 Mdot
.69264 .1496 Mdot
.53176 .9187 Mdot
.53922 .91285 Mdot
.52065 .92486 Mdot
.72279 .15905 Mdot
.70687 .15345 Mdot
.65977 .14603 Mdot
.54746 .90212 Mdot
.5488 .8973 Mdot
.54421 .90734 Mdot
.6804 .14741 Mdot
.66957 .14633 Mdot
.81938 .22497 Mdot
.34042 .93064 Mdot
.4066 .94155 Mdot
.24626 .89081 Mdot
.89353 .32649 Mdot
.85486 .26528 Mdot
.74163 .16768 Mdot
.48314 .93693 Mdot
.50532 .93106 Mdot
.45189 .94132 Mdot
.78898 .19814 Mdot
.76318 .17989 Mdot
.60389 .16955 Mdot
.52563 .86347 Mdot
.51903 .85881 Mdot
.53125 .86787 Mdot
.61397 .16093 Mdot
.60879 .16492 Mdot
.58814 .18701 Mdot
.50398 .84803 Mdot
.4952 .84172 Mdot
.51179 .8537 Mdot
.59898 .17478 Mdot
.5936 .18051 Mdot
.63657 .14938 Mdot
.54837 .88834 Mdot
.54642 .8842 Mdot
.54915 .8927 Mdot
.65135 .14664 Mdot
.64353 .14771 Mdot
.61901 .1573 Mdot
.5405 .87608 Mdot
.53614 .87206 Mdot
.54391 .88011 Mdot
.63039 .15162 Mdot
.6244 .15417 Mdot
.49224 .15579 Mdot
.58209 .8036 Mdot
.5019 .14885 Mdot
.5885 .81158 Mdot
.57482 .79484 Mdot
.47096 .17268 Mdot
.48184 .16368 Mdot
.60554 .8307 Mdot
.51854 .13715 Mdot
.52583 .13215 Mdot
.51054 .14267 Mdot
.59448 .81871 Mdot
.60027 .825 Mdot
.42446 .22186 Mdot
.52787 .73675 Mdot
.53861 .75022 Mdot
.51654 .72257 Mdot
.40315 .25432 Mdot
.41348 .23752 Mdot
.45963 .18292 Mdot
.55838 .77439 Mdot
.56693 .78512 Mdot
.54887 .76277 Mdot
.43598 .20749 Mdot
.44784 .19451 Mdot
.6642 .8561 Mdot
.55204 .0966 Mdot
.55064 .09149 Mdot
.55237 .10147 Mdot
.647 .85438 Mdot
.65528 .85548 Mdot
.69909 .85212 Mdot
.54193 .08006 Mdot
.53392 .07381 Mdot
.54751 .086 Mdot
.67437 .85586 Mdot
.6861 .85452 Mdot
.62148 .844 Mdot
.53751 .12306 Mdot
.54233 .11884 Mdot
.53214 .1275 Mdot
.61072 .83575 Mdot
.61617 .84009 Mdot
.63983 .85253 Mdot
.54905 .11035 Mdot
.55122 .106 Mdot
.54626 .11464 Mdot
.62706 .8474 Mdot
.63333 .85015 Mdot
.40807 .50257 Mdot
.36067 .50092 Mdot
.40854 .51226 Mdot
.36064 .4907 Mdot
.36064 .51113 Mdot
.40884 .48316 Mdot
.40816 .49288 Mdot
.36031 .45878 Mdot
.41129 .53223 Mdot
.41359 .54266 Mdot
.4096 .52211 Mdot
.36056 .48032 Mdot
.36045 .46972 Mdot
.41774 .44148 Mdot
.36045 .56736 Mdot
.36033 .55501 Mdot
.36078 .58046 Mdot
.42656 .41775 Mdot
.42176 .4299 Mdot
.41009 .47326 Mdot
.36047 .53227 Mdot
.36057 .52156 Mdot
.36036 .54335 Mdot
.41451 .4525 Mdot
.41198 .46305 Mdot
.37348 .32684 Mdot
.46145 .64628 Mdot
.47142 .6616 Mdot
.45223 .63131 Mdot
.36588 .36196 Mdot
.36912 .34473 Mdot
.39383 .27199 Mdot
.49348 .69253 Mdot
.50493 .70778 Mdot
.48221 .67708 Mdot
.37903 .30855 Mdot
.38577 .29018 Mdot
.36051 .42224 Mdot
.42037 .56499 Mdot
.4249 .57698 Mdot
.41659 .55356 Mdot
.36024 .44726 Mdot
.36029 .4351 Mdot
.36361 .37835 Mdot
.4367 .60291 Mdot
.44396 .61682 Mdot
.43033 .58962 Mdot
.36103 .40854 Mdot
.36203 .3939 Mdot
.59314 .18172 Mdot
.50784 .85457 Mdot
.58675 .18909 Mdot
.49869 .84813 Mdot
.51631 .86028 Mdot
.60421 .1691 Mdot
.59886 .17506 Mdot
.46675 .82302 Mdot
.57263 .20666 Mdot
.56424 .21677 Mdot
.58003 .19742 Mdot
.48887 .84082 Mdot
.47812 .83251 Mdot
.62584 .15175 Mdot
.54532 .88317 Mdot
.54134 .87893 Mdot
.5486 .88737 Mdot
.63808 .14622 Mdot
.6318 .14877 Mdot
.60966 .16393 Mdot
.53033 .87023 Mdot
.52384 .86544 Mdot
.53624 .87467 Mdot
.62013 .1552 Mdot
.61491 .15931 Mdot
.37573 .68222 Mdot
.47737 .32801 Mdot
.4667 .34361 Mdot
.48854 .31232 Mdot
.3898 .71943 Mdot
.38209 .70085 Mdot
.36399 .62894 Mdot
.44801 .37392 Mdot
.44002 .38833 Mdot
.45687 .35898 Mdot
.37074 .66389 Mdot
.36686 .64606 Mdot
.43128 .78633 Mdot
.54542 .24008 Mdot
.53471 .2531 Mdot
.5552 .22793 Mdot
.45507 .81221 Mdot
.44323 .79999 Mdot
.39874 .73757 Mdot
.51199 .28164 Mdot
.50017 .29679 Mdot
.5235 .267 Mdot
.41963 .77126 Mdot
.4087 .75493 Mdot
.94156 .53974 Mdot
.08239 .28424 Mdot
.037 .53817 Mdot
.20137 .13106 Mdot
.87605 .71583 Mdot
.91556 .64188 Mdot
.86465 .26648 Mdot
.23586 .8921 Mdot
.33713 .93665 Mdot
.1135 .77032 Mdot
.93676 .42729 Mdot
.90485 .33244 Mdot
.75196 .83171 Mdot
.50453 .06053 Mdot
.47945 .05435 Mdot
.52171 .06718 Mdot
.71437 .8478 Mdot
.73163 .84142 Mdot
.83732 .76458 Mdot
.39009 .05194 Mdot
.3114 .06999 Mdot
.44346 .05019 Mdot
.77562 .81755 Mdot
.80396 .79627 Mdot
.67172 .14229 Mdot
.54877 .91127 Mdot
.55164 .90586 Mdot
.54411 .91702 Mdot
.69537 .14529 Mdot
.68277 .1432 Mdot
.64527 .14434 Mdot
.55235 .89616 Mdot
.55112 .89164 Mdot
.55256 .90088 Mdot
.66168 .14215 Mdot
.65309 .14297 Mdot
.76853 .17613 Mdot
.45542 .94771 Mdot
.48775 .94276 Mdot
.4077 .9483 Mdot
.82733 .22339 Mdot
.79552 .19518 Mdot
.71004 .14909 Mdot
.52588 .9297 Mdot
.53691 .92316 Mdot
.51045 .93635 Mdot
.74608 .16353 Mdot
.72652 .15472 Mdot
.59905 .82375 Mdot
.50995 .14427 Mdot
.60419 .82936 Mdot
.51778 .13884 Mdot
.50154 .15036 Mdot
.58753 .8106 Mdot
.59339 .81757 Mdot
.53636 .12492 Mdot
.61464 .83862 Mdot
.61981 .84251 Mdot
.6093 .83432 Mdot
.52491 .1339 Mdot
.53102 .12929 Mdot
.55825 .77426 Mdot
.44891 .19492 Mdot
.4604 .18365 Mdot
.43734 .20753 Mdot
.53909 .7507 Mdot
.54903 .76294 Mdot
.58129 .80278 Mdot
.48201 .16489 Mdot
.49215 .15717 Mdot
.47142 .17368 Mdot
.56654 .78473 Mdot
.57422 .79422 Mdot
.54179 .08283 Mdot
.68249 .85354 Mdot
.695 .85141 Mdot
.6713 .85462 Mdot
.54979 .09394 Mdot
.54682 .08857 Mdot
.5074 .0638 Mdot
.72614 .84174 Mdot
.74561 .83285 Mdot
.70969 .84748 Mdot
.5344 .07676 Mdot
.5231 .07031 Mdot
.54775 .11237 Mdot
.63142 .84861 Mdot
.63772 .85103 Mdot
.62531 .84586 Mdot
.54112 .12075 Mdot
.54493 .11659 Mdot
.55095 .09891 Mdot
.65276 .85406 Mdot
.66136 .85479 Mdot
.64475 .85287 Mdot
.54999 .10812 Mdot
.55112 .10365 Mdot
.36126 .4662 Mdot
.41065 .52544 Mdot
.36114 .45529 Mdot
.41251 .53551 Mdot
.40942 .51569 Mdot
.36147 .48702 Mdot
.36138 .47676 Mdot
.42205 .56817 Mdot
.3612 .43164 Mdot
.36151 .41876 Mdot
.3611 .4438 Mdot
.41498 .5459 Mdot
.41813 .55676 Mdot
.36136 .52768 Mdot
.41166 .46748 Mdot
.41006 .47739 Mdot
.41385 .45727 Mdot
.3612 .54963 Mdot
.36125 .53839 Mdot
.36152 .49707 Mdot
.40863 .49658 Mdot
.40874 .50611 Mdot
.40907 .48706 Mdot
.36146 .51731 Mdot
.36151 .50713 Mdot
.49515 .69437 Mdot
.38775 .28801 Mdot
.3958 .27025 Mdot
.38094 .30602 Mdot
.47343 .664 Mdot
.48406 .67922 Mdot
.52867 .73757 Mdot
.41528 .2367 Mdot
.42607 .22149 Mdot
.40507 .25304 Mdot
.50634 .70929 Mdot
.51764 .72374 Mdot
.43873 .60594 Mdot
.36323 .39048 Mdot
.36496 .37499 Mdot
.36211 .40508 Mdot
.42673 .58013 Mdot
.43225 .59272 Mdot
.46354 .6489 Mdot
.37076 .34166 Mdot
.37526 .32401 Mdot
.36739 .3587 Mdot
.44605 .61976 Mdot
.45433 .63411 Mdot
.48413 .83602 Mdot
.58313 .19361 Mdot
.47359 .82739 Mdot
.57607 .20205 Mdot
.58934 .18592 Mdot
.5036 .85025 Mdot
.49422 .84359 Mdot
.551 .2331 Mdot
.45079 .80646 Mdot
.43907 .79396 Mdot
.46242 .81758 Mdot
.56851 .21147 Mdot
.56026 .22185 Mdot
.53274 .87089 Mdot
.61634 .15849 Mdot
.61097 .16266 Mdot
.62163 .15475 Mdot
.54239 .87937 Mdot
.53783 .87522 Mdot
.51199 .85619 Mdot
.60074 .173 Mdot
.59509 .17904 Mdot
.60591 .16752 Mdot
.52664 .86637 Mdot
.51963 .86153 Mdot
.45489 .36292 Mdot
.37047 .65888 Mdot
.36697 .64156 Mdot
.37516 .6768 Mdot
.47449 .33261 Mdot
.46435 .34791 Mdot
.43223 .40497 Mdot
.36259 .60921 Mdot
.36142 .59436 Mdot
.36441 .62498 Mdot
.4464 .3775 Mdot
.43888 .39153 Mdot
.51944 .27232 Mdot
.41653 .76489 Mdot
.40603 .74851 Mdot
.42764 .78009 Mdot
.5411 .24532 Mdot
.53062 .25846 Mdot
.48532 .3172 Mdot
.38818 .71329 Mdot
.38108 .69505 Mdot
.39649 .73121 Mdot
.50804 .28686 Mdot
.49666 .3019 Mdot
.20082 .86439 Mdot
.91326 .3575 Mdot
.87431 .2842 Mdot
.93889 .4583 Mdot
.03872 .46338 Mdot
.08396 .71302 Mdot
.4409 .94645 Mdot
.80303 .20404 Mdot
.77493 .1827 Mdot
.83607 .23571 Mdot
.30949 .92566 Mdot
.38765 .94425 Mdot
.40387 .05288 Mdot
.79489 .80092 Mdot
.82649 .77262 Mdot
.76801 .8201 Mdot
.48443 .05759 Mdot
.45185 .05295 Mdot
.11251 .23358 Mdot
.90323 .66356 Mdot
.93444 .56941 Mdot
.86354 .7294 Mdot
.3333 .06537 Mdot
.23272 .11111 Mdot
.55159 .89648 Mdot
.65529 .1439 Mdot
.64702 .14489 Mdot
.66419 .14341 Mdot
.54944 .90634 Mdot
.55105 .9013 Mdot
.54612 .88348 Mdot
.63338 .14887 Mdot
.62714 .15149 Mdot
.63989 .14663 Mdot
.55063 .89201 Mdot
.54869 .88772 Mdot
.51965 .93024 Mdot
.73127 .15864 Mdot
.71413 .15216 Mdot
.75147 .16847 Mdot
.47699 .94264 Mdot
.50227 .93671 Mdot
.5461 .91176 Mdot
.686 .14521 Mdot
.67432 .14375 Mdot
.69893 .14772 Mdot
.53208 .92375 Mdot
.54031 .91761 Mdot
.60504 .16983 Mdot
.51512 .87115 Mdot
.59739 .17865 Mdot
.50357 .86381 Mdot
.52551 .87754 Mdot
.61849 .15528 Mdot
.61193 .16204 Mdot
.46303 .83354 Mdot
.57916 .19985 Mdot
.56853 .21245 Mdot
.5887 .18858 Mdot
.49101 .85525 Mdot
.47749 .84524 Mdot
.6448 .13657 Mdot
.55911 .90297 Mdot
.55472 .89818 Mdot
.56202 .90796 Mdot
.66084 .1316 Mdot
.65254 .13379 Mdot
.62503 .1495 Mdot
.54232 .88851 Mdot
.5346 .88325 Mdot
.54905 .89343 Mdot
.63776 .1401 Mdot
.63133 .14444 Mdot
.35961 .65784 Mdot
.45945 .34846 Mdot
.44795 .36627 Mdot
.47214 .33015 Mdot
.37201 .70182 Mdot
.3649 .67952 Mdot
.35196 .59953 Mdot
.42932 .39962 Mdot
.42202 .41499 Mdot
.43793 .38337 Mdot
.35592 .63718 Mdot
.35346 .61771 Mdot
.41817 .78663 Mdot
.54343 .24146 Mdot
.52952 .25781 Mdot
.55659 .22633 Mdot
.448 .81992 Mdot
.43288 .80428 Mdot
.38103 .72424 Mdot
.50029 .29316 Mdot
.48581 .3116 Mdot
.51505 .27515 Mdot
.4044 .76716 Mdot
.39191 .74621 Mdot
.40021 .02955 Mdot
.82175 .80823 Mdot
.30785 .04873 Mdot
.85978 .77245 Mdot
.78985 .83137 Mdot
.49841 .03663 Mdot
.45996 .03005 Mdot
.96922 .50589 Mdot
.04329 .33152 Mdot
.03475 .6379 Mdot
.1749 .12637 Mdot
.904 .71556 Mdot
.94758 .62655 Mdot
.55856 .0681 Mdot
.69182 .8679 Mdot
.70605 .86587 Mdot
.67978 .86835 Mdot
.56385 .08073 Mdot
.5622 .07465 Mdot
.52407 .04485 Mdot
.74111 .85605 Mdot
.76362 .84615 Mdot
.72228 .86217 Mdot
.55178 .06091 Mdot
.54066 .05307 Mdot
.70763 .13259 Mdot
.55118 .93865 Mdot
.55818 .93145 Mdot
.53958 .94652 Mdot
.74348 .14314 Mdot
.72437 .1367 Mdot
.67025 .13023 Mdot
.56414 .91871 Mdot
.56381 .91316 Mdot
.56235 .9248 Mdot
.69351 .13056 Mdot
.68125 .12993 Mdot
.86406 .23056 Mdot
.2961 .94568 Mdot
.39254 .96845 Mdot
.1599 .85835 Mdot
.95124 .38286 Mdot
.90877 .29033 Mdot
.76619 .15339 Mdot
.49554 .96288 Mdot
.52204 .95478 Mdot
.45509 .96912 Mdot
.82544 .19313 Mdot
.79309 .16902 Mdot
.63082 .85378 Mdot
.54138 .11023 Mdot
.63733 .85833 Mdot
.5483 .10516 Mdot
.53315 .11563 Mdot
.61771 .84247 Mdot
.62426 .84856 Mdot
.56246 .09048 Mdot
.65196 .8651 Mdot
.66038 .86739 Mdot
.64422 .86214 Mdot
.55424 .1003 Mdot
.55905 .09547 Mdot
.58686 .80772 Mdot
.47398 .15543 Mdot
.48799 .14491 Mdot
.45915 .16772 Mdot
.56587 .78295 Mdot
.57699 .79599 Mdot
.61106 .83536 Mdot
.51277 .12821 Mdot
.52356 .12154 Mdot
.50089 .13592 Mdot
.59579 .81812 Mdot
.60389 .82725 Mdot
.46042 .02805 Mdot
.79202 .83213 Mdot
.82399 .80898 Mdot
.765 .84745 Mdot
.52533 .04319 Mdot
.49988 .03484 Mdot
.16918 .12883 Mdot
.90779 .71382 Mdot
.95138 .62285 Mdot
.86281 .77227 Mdot
.39966 .02753 Mdot
.30501 .04754 Mdot
.56375 .07354 Mdot
.6804 .86957 Mdot
.69285 .86898 Mdot
.66966 .86898 Mdot
.56435 .08519 Mdot
.56496 .0796 Mdot
.54244 .05165 Mdot
.72331 .86347 Mdot
.74262 .85714 Mdot
.70686 .8672 Mdot
.55988 .06682 Mdot
.55319 .05954 Mdot
.35118 .47663 Mdot
.40258 .51474 Mdot
.351 .46381 Mdot
.40411 .52649 Mdot
.40188 .50321 Mdot
.35136 .50117 Mdot
.35131 .48902 Mdot
.41439 .56481 Mdot
.35075 .43611 Mdot
.35091 .42081 Mdot
.35083 .45036 Mdot
.40654 .53865 Mdot
.40995 .5514 Mdot
.35088 .55251 Mdot
.41127 .44308 Mdot
.40754 .45598 Mdot
.41604 .42946 Mdot
.35117 .58265 Mdot
.35087 .56702 Mdot
.3513 .51339 Mdot
.40298 .48016 Mdot
.40202 .49172 Mdot
.4048 .46831 Mdot
.35101 .53887 Mdot
.35118 .52589 Mdot
.511 .7186 Mdot
.38844 .25817 Mdot
.40062 .23672 Mdot
.37801 .28048 Mdot
.48179 .68171 Mdot
.49612 .70033 Mdot
.55338 .76865 Mdot
.42875 .19833 Mdot
.44392 .182 Mdot
.41418 .21664 Mdot
.52574 .73626 Mdot
.53994 .75303 Mdot
.43513 .61023 Mdot
.35275 .38644 Mdot
.35489 .36727 Mdot
.3515 .40429 Mdot
.41998 .57904 Mdot
.42689 .59419 Mdot
.4683 .66311 Mdot
.36295 .32534 Mdot
.36952 .30306 Mdot
.35818 .34685 Mdot
.44475 .62715 Mdot
.45584 .64484 Mdot
.5629 .91787 Mdot
.68109 .13111 Mdot
.56289 .91233 Mdot
.67019 .13135 Mdot
.69327 .13181 Mdot
.55651 .93052 Mdot
.56073 .92396 Mdot
.64476 .13729 Mdot
.55856 .90224 Mdot
.55461 .89746 Mdot
.56128 .90718 Mdot
.66062 .13252 Mdot
.65233 .13458 Mdot
.49249 .96143 Mdot
.79251 .17086 Mdot
.76593 .15526 Mdot
.8248 .19522 Mdot
.3893 .96604 Mdot
.45199 .96729 Mdot
.54918 .93767 Mdot
.72412 .13821 Mdot
.70755 .13406 Mdot
.74314 .14475 Mdot
.51955 .95353 Mdot
.5372 .94546 Mdot
.58095 .19757 Mdot
.48099 .84662 Mdot
.46707 .83558 Mdot
.49394 .85608 Mdot
.59831 .17739 Mdot
.59005 .18685 Mdot
.547 .23738 Mdot
.43767 .80797 Mdot
.42311 .79122 Mdot
.45248 .82274 Mdot
.57087 .2096 Mdot
.55958 .22289 Mdot
.62504 .14958 Mdot
.54285 .88797 Mdot
.53548 .88286 Mdot
.54929 .89277 Mdot
.63762 .14059 Mdot
.63119 .14472 Mdot
.60558 .16895 Mdot
.51706 .87118 Mdot
.50601 .86419 Mdot
.52694 .87732 Mdot
.6187 .15512 Mdot
.61223 .16153 Mdot
.65946 .86548 Mdot
.56076 .09228 Mdot
.66875 .86684 Mdot
.56241 .08711 Mdot
.55802 .09724 Mdot
.64354 .86065 Mdot
.65123 .86334 Mdot
.55673 .06902 Mdot
.6917 .86655 Mdot
.7056 .8646 Mdot
.67961 .86715 Mdot
.56267 .08161 Mdot
.56084 .07558 Mdot
.61766 .84256 Mdot
.52599 .12225 Mdot
.53469 .1167 Mdot
.51609 .12843 Mdot
.6047 .8286 Mdot
.61134 .83605 Mdot
.63662 .8572 Mdot
.54842 .10669 Mdot
.55382 .10198 Mdot
.54204 .11154 Mdot
.62404 .84812 Mdot
.63028 .85299 Mdot
.04477 .33291 Mdot
.94529 .6258 Mdot
.96673 .50609 Mdot
.90236 .71392 Mdot
.30537 .05178 Mdot
.17415 .1297 Mdot
.29314 .94233 Mdot
.90741 .29247 Mdot
.86321 .23281 Mdot
.94926 .38467 Mdot
.03589 .63502 Mdot
.15827 .85397 Mdot
.52152 .0461 Mdot
.74077 .85442 Mdot
.76303 .84452 Mdot
.72204 .86063 Mdot
.54986 .06193 Mdot
.53852 .05419 Mdot
.39726 .03188 Mdot
.82085 .80639 Mdot
.8585 .77072 Mdot
.78928 .82951 Mdot
.49588 .03813 Mdot
.45705 .03187 Mdot
.52916 .74026 Mdot
.40467 .23207 Mdot
.54303 .75643 Mdot
.41834 .21278 Mdot
.39236 .2528 Mdot
.50042 .70524 Mdot
.51488 .7231 Mdot
.4626 .16614 Mdot
.5677 .7853 Mdot
.57831 .79783 Mdot
.55597 .77146 Mdot
.4329 .19526 Mdot
.44783 .17971 Mdot
.44872 .63289 Mdot
.35671 .36066 Mdot
.36046 .34017 Mdot
.35427 .37994 Mdot
.43017 .59982 Mdot
.43872 .61594 Mdot
.4861 .687 Mdot
.37282 .29665 Mdot
.38171 .2745 Mdot
.36576 .3187 Mdot
.46005 .65053 Mdot
.47255 .66865 Mdot
.55404 .10126 Mdot
.63682 .8577 Mdot
.64391 .86124 Mdot
.63025 .85338 Mdot
.54152 .11098 Mdot
.5484 .10606 Mdot
.56354 .08631 Mdot
.65958 .86648 Mdot
.66916 .8678 Mdot
.65133 .86422 Mdot
.55828 .09641 Mdot
.56152 .09147 Mdot
.50287 .13585 Mdot
.59648 .81897 Mdot
.60402 .82784 Mdot
.58798 .80899 Mdot
.47677 .15444 Mdot
.49026 .14442 Mdot
.53358 .11623 Mdot
.61773 .84232 Mdot
.62404 .84823 Mdot
.611 .83561 Mdot
.5144 .12847 Mdot
.52465 .12202 Mdot
.43318 .39308 Mdot
.35522 .6253 Mdot
.35337 .60686 Mdot
.35812 .64496 Mdot
.45257 .35958 Mdot
.44219 .37672 Mdot
.41383 .43701 Mdot
.35182 .5738 Mdot
.35172 .55905 Mdot
.3523 .5897 Mdot
.42545 .40859 Mdot
.41903 .42323 Mdot
.50522 .28739 Mdot
.39661 .75251 Mdot
.38536 .73122 Mdot
.4093 .77265 Mdot
.53363 .25311 Mdot
.51963 .26987 Mdot
.46435 .34183 Mdot
.36817 .68727 Mdot
.36232 .66569 Mdot
.37584 .70929 Mdot
.49087 .30543 Mdot
.47718 .32368 Mdot
.40839 .54389 Mdot
.35178 .45822 Mdot
.35162 .44471 Mdot
.35196 .47107 Mdot
.4037 .51992 Mdot
.40559 .53167 Mdot
.42294 .58459 Mdot
.35196 .41471 Mdot
.35278 .39796 Mdot
.35164 .43026 Mdot
.41213 .55668 Mdot
.41693 .57021 Mdot
.40433 .47432 Mdot
.35194 .53213 Mdot
.35209 .51961 Mdot
.35179 .54521 Mdot
.4097 .45004 Mdot
.40655 .46245 Mdot
.40266 .50849 Mdot
.35219 .49553 Mdot
.35211 .48345 Mdot
.35219 .50749 Mdot
.40297 .48584 Mdot
.40241 .49719 Mdot
.34961 .48691 Mdot
.40082 .50496 Mdot
.34946 .47408 Mdot
.40168 .51685 Mdot
.40084 .49316 Mdot
.34963 .51211 Mdot
.34967 .49951 Mdot
.40995 .55495 Mdot
.34907 .44678 Mdot
.34905 .43178 Mdot
.34925 .4608 Mdot
.40345 .52905 Mdot
.40619 .54169 Mdot
.34917 .56726 Mdot
.41514 .4291 Mdot
.41015 .44316 Mdot
.42137 .41419 Mdot
.35026 .60083 Mdot
.34947 .58341 Mdot
.34949 .52492 Mdot
.40357 .46905 Mdot
.40175 .48125 Mdot
.40634 .45642 Mdot
.34916 .55225 Mdot
.3493 .5382 Mdot
.50371 .7103 Mdot
.38132 .2682 Mdot
.39303 .24546 Mdot
.37159 .29152 Mdot
.47384 .67198 Mdot
.48839 .69122 Mdot
.54878 .76338 Mdot
.4213 .20404 Mdot
.43698 .18621 Mdot
.40647 .22391 Mdot
.51913 .72891 Mdot
.53425 .74671 Mdot
.42868 .59983 Mdot
.3501 .39816 Mdot
.3516 .37932 Mdot
.34932 .41562 Mdot
.41489 .569 Mdot
.42111 .58394 Mdot
.46045 .65295 Mdot
.35817 .33744 Mdot
.3639 .31479 Mdot
.35418 .35905 Mdot
.43771 .61668 Mdot
.44835 .63444 Mdot
.61733 .15665 Mdot
.53133 .88329 Mdot
.61028 .16419 Mdot
.52114 .87707 Mdot
.54006 .88891 Mdot
.63071 .14457 Mdot
.62419 .1502 Mdot
.48295 .85202 Mdot
.59425 .18273 Mdot
.58453 .19382 Mdot
.60272 .17287 Mdot
.50958 .86998 Mdot
.49679 .86173 Mdot
.66021 .13005 Mdot
.56545 .91414 Mdot
.56305 .9089 Mdot
.56649 .91969 Mdot
.68019 .12762 Mdot
.66942 .1283 Mdot
.63724 .13973 Mdot
.55417 .899 Mdot
.54768 .89408 Mdot
.55938 .90387 Mdot
.65197 .13258 Mdot
.64424 .13573 Mdot
.36425 .68398 Mdot
.47405 .3264 Mdot
.46068 .34535 Mdot
.48859 .30728 Mdot
.38158 .73026 Mdot
.37192 .70711 Mdot
.35181 .61964 Mdot
.43803 .3815 Mdot
.42895 .39833 Mdot
.44866 .36379 Mdot
.3585 .66143 Mdot
.35445 .63989 Mdot
.43698 .81166 Mdot
.56179 .22029 Mdot
.54853 .2356 Mdot
.57376 .20634 Mdot
.46824 .84059 Mdot
.45278 .82719 Mdot
.39322 .75284 Mdot
.51901 .26979 Mdot
.50373 .28829 Mdot
.53406 .25213 Mdot
.4214 .79396 Mdot
.40663 .77424 Mdot
.74051 .13699 Mdot
.53324 .95668 Mdot
.72161 .13134 Mdot
.54883 .94803 Mdot
.50978 .96547 Mdot
.78917 .16013 Mdot
.76256 .1459 Mdot
.56708 .9261 Mdot
.69159 .12668 Mdot
.67954 .12655 Mdot
.70533 .12804 Mdot
.55843 .94006 Mdot
.56406 .93279 Mdot
.95004 .35782 Mdot
.04963 .70078 Mdot
.19346 .8952 Mdot
.02381 .38351 Mdot
.9609 .60851 Mdot
.97619 .47989 Mdot
.82064 .1817 Mdot
.41638 .97619 Mdot
.4731 .97327 Mdot
.32686 .96226 Mdot
.90411 .27094 Mdot
.85902 .21604 Mdot
.51851 .87657 Mdot
.61657 .15765 Mdot
.60916 .1656 Mdot
.6236 .15081 Mdot
.53855 .88895 Mdot
.52924 .88312 Mdot
.47834 .8499 Mdot
.59227 .18523 Mdot
.58202 .19697 Mdot
.6012 .17478 Mdot
.50634 .86907 Mdot
.49288 .86028 Mdot
.56333 .90928 Mdot
.65971 .12935 Mdot
.65156 .13209 Mdot
.66886 .12742 Mdot
.56758 .92006 Mdot
.56615 .91452 Mdot
.54672 .89427 Mdot
.63683 .13971 Mdot
.63021 .14483 Mdot
.6439 .13549 Mdot
.55928 .90422 Mdot
.55368 .89928 Mdot
.63193 .85495 Mdot
.54118 .10847 Mdot
.54868 .1032 Mdot
.53223 .1141 Mdot
.6183 .84283 Mdot
.62512 .84937 Mdot
.65345 .86705 Mdot
.56035 .09326 Mdot
.56421 .08819 Mdot
.5551 .0982 Mdot
.63863 .85978 Mdot
.64564 .86385 Mdot
.58517 .80543 Mdot
.46863 .15707 Mdot
.48357 .14557 Mdot
.45294 .17056 Mdot
.56228 .77875 Mdot
.57438 .79279 Mdot
.61126 .83521 Mdot
.51024 .12749 Mdot
.52187 .12035 Mdot
.49743 .1358 Mdot
.59488 .81663 Mdot
.60359 .82649 Mdot
.79754 .83295 Mdot
.50052 .0308 Mdot
.45905 .02381 Mdot
.52722 .03952 Mdot
.74642 .8591 Mdot
.76958 .84898 Mdot
.91779 .70621 Mdot
.29275 .0477 Mdot
.14891 .14296 Mdot
.39441 .02387 Mdot
.83086 .80838 Mdot
.87134 .76912 Mdot
.68243 .87188 Mdot
.56719 .07707 Mdot
.5662 .07088 Mdot
.56641 .08279 Mdot
.66211 .86945 Mdot
.67158 .87115 Mdot
.72641 .86572 Mdot
.5557 .05647 Mdot
.54476 .0483 Mdot
.5626 .06402 Mdot
.69521 .8713 Mdot
.70963 .86946 Mdot
.46655 .3375 Mdot
.36802 .6932 Mdot
.45405 .35591 Mdot
.36162 .67074 Mdot
.3764 .71603 Mdot
.4946 .2999 Mdot
.48016 .31873 Mdot
.35177 .60939 Mdot
.43347 .39069 Mdot
.42529 .40678 Mdot
.44303 .3737 Mdot
.357 .64912 Mdot
.35381 .6286 Mdot
.55272 .23061 Mdot
.44266 .81616 Mdot
.42739 .79953 Mdot
.45805 .83068 Mdot
.57683 .20288 Mdot
.5654 .21607 Mdot
.50958 .28137 Mdot
.39898 .76045 Mdot
.38677 .73864 Mdot
.41269 .78088 Mdot
.5391 .24649 Mdot
.52464 .2635 Mdot
.3504 .48182 Mdot
.40162 .50983 Mdot
.40282 .52164 Mdot
.40128 .49818 Mdot
.3505 .50661 Mdot
.3505 .49429 Mdot
.3499 .44151 Mdot
.408 .54649 Mdot
.41211 .55982 Mdot
.40492 .53379 Mdot
.35023 .469 Mdot
.35004 .45563 Mdot
.35 .55977 Mdot
.41305 .43619 Mdot
.40873 .44964 Mdot
.41852 .42194 Mdot
.35062 .59154 Mdot
.35011 .57504 Mdot
.3504 .5191 Mdot
.40317 .47465 Mdot
.4018 .48649 Mdot
.40545 .46243 Mdot
.35009 .54548 Mdot
.35025 .53199 Mdot
.54256 .94719 Mdot
.74296 .14126 Mdot
.55335 .93932 Mdot
.72356 .13482 Mdot
.76541 .15104 Mdot
.49978 .96395 Mdot
.52557 .9556 Mdot
.68067 .12865 Mdot
.56396 .92536 Mdot
.56517 .91932 Mdot
.56033 .93199 Mdot
.70729 .13118 Mdot
.69312 .12926 Mdot
.16822 .8689 Mdot
.9517 .37648 Mdot
.9082 .28519 Mdot
.97176 .49955 Mdot
.03788 .34275 Mdot
.03735 .65404 Mdot
.46034 .9707 Mdot
.8244 .18968 Mdot
.7923 .16631 Mdot
.86328 .22657 Mdot
.30452 .95094 Mdot
.39947 .97116 Mdot
.61139 .16302 Mdot
.52337 .87745 Mdot
.51254 .87072 Mdot
.53308 .88338 Mdot
.62451 .14965 Mdot
.61808 .15586 Mdot
.58689 .19099 Mdot
.48739 .85385 Mdot
.47309 .84318 Mdot
.5006 .86293 Mdot
.604 .17115 Mdot
.59583 .18041 Mdot
.65214 .13302 Mdot
.56279 .90845 Mdot
.55922 .90351 Mdot
.56477 .9137 Mdot
.6701 .12931 Mdot
.66073 .13083 Mdot
.6311 .1444 Mdot
.54861 .89381 Mdot
.54154 .88875 Mdot
.55441 .89869 Mdot
.64462 .13609 Mdot
.63776 .13992 Mdot
.50021 .03283 Mdot
.76728 .84821 Mdot
.45993 .02594 Mdot
.79463 .83266 Mdot
.74447 .85814 Mdot
.54358 .04998 Mdot
.52644 .04139 Mdot
.91269 .71024 Mdot
.2994 .04737 Mdot
.15946 .13523 Mdot
.39738 .02565 Mdot
.82737 .80873 Mdot
.86701 .77078 Mdot
.56602 .07833 Mdot
.67068 .87002 Mdot
.68136 .87074 Mdot
.66124 .86841 Mdot
.56336 .08935 Mdot
.56547 .08402 Mdot
.55443 .05801 Mdot
.70829 .86829 Mdot
.72478 .86464 Mdot
.69395 .87018 Mdot
.56492 .0722 Mdot
.56138 .06546 Mdot
.74175 .13909 Mdot
.52937 .95619 Mdot
.54575 .94764 Mdot
.50486 .96476 Mdot
.79082 .16324 Mdot
.76399 .14843 Mdot
.69239 .12795 Mdot
.56212 .93244 Mdot
.56559 .92574 Mdot
.55598 .9397 Mdot
.72269 .13312 Mdot
.70626 .12954 Mdot
.95102 .36722 Mdot
.04277 .67701 Mdot
.18046 .88225 Mdot
.0307 .36194 Mdot
.95613 .61596 Mdot
.97412 .49026 Mdot
.82257 .18567 Mdot
.40776 .97377 Mdot
.4667 .97206 Mdot
.31542 .95672 Mdot
.90626 .27807 Mdot
.86118 .22125 Mdot
.3848 .26318 Mdot
.49223 .69577 Mdot
.50732 .71446 Mdot
.47782 .67682 Mdot
.3666 .30897 Mdot
.37469 .28602 Mdot
.42503 .20114 Mdot
.53716 .74987 Mdot
.55106 .76604 Mdot
.52249 .73259 Mdot
.3968 .24106 Mdot
.41036 .22023 Mdot
.35137 .3924 Mdot
.42391 .58899 Mdot
.43184 .60496 Mdot
.41736 .57395 Mdot
.34994 .42639 Mdot
.35036 .41006 Mdot
.36048 .33144 Mdot
.452 .63959 Mdot
.46433 .65799 Mdot
.44117 .62186 Mdot
.3532 .37336 Mdot
.35613 .35299 Mdot
.54132 .10936 Mdot
.62462 .849 Mdot
.63133 .85438 Mdot
.61805 .84263 Mdot
.52266 .12093 Mdot
.53263 .11486 Mdot
.55962 .09435 Mdot
.64496 .86297 Mdot
.65258 .86613 Mdot
.63803 .85902 Mdot
.54858 .10421 Mdot
.55462 .09924 Mdot
.47127 .15623 Mdot
.57576 .79439 Mdot
.58608 .80657 Mdot
.56407 .78088 Mdot
.44042 .18407 Mdot
.45599 .16911 Mdot
.51151 .12785 Mdot
.6037 .82689 Mdot
.61122 .83526 Mdot
.59529 .81741 Mdot
.48581 .14523 Mdot
.49928 .13587 Mdot
.55772 .08147 Mdot
.67315 .86362 Mdot
.55442 .07539 Mdot
.68438 .86329 Mdot
.66332 .86298 Mdot
.55911 .09242 Mdot
.55913 .08712 Mdot
.72957 .85329 Mdot
.53862 .06157 Mdot
.52424 .05403 Mdot
.54814 .0687 Mdot
.69755 .86159 Mdot
.71228 .85855 Mdot
.54599 .11119 Mdot
.62658 .8497 Mdot
.63277 .85363 Mdot
.62079 .84505 Mdot
.53301 .12072 Mdot
.53995 .11583 Mdot
.55742 .09732 Mdot
.64648 .85963 Mdot
.6543 .86172 Mdot
.63948 .85687 Mdot
.55089 .10663 Mdot
.55457 .102 Mdot
.92515 .34131 Mdot
.08584 .74724 Mdot
.22 .8969 Mdot
.02485 .4749 Mdot
.95265 .56945 Mdot
.9558 .44834 Mdot
.80677 .18833 Mdot
.41293 .96153 Mdot
.4645 .95978 Mdot
.33462 .94936 Mdot
.88201 .26673 Mdot
.84125 .21885 Mdot
.80146 .81401 Mdot
.4708 .03979 Mdot
.42281 .03688 Mdot
.50261 .04642 Mdot
.74972 .84509 Mdot
.77324 .83281 Mdot
.91836 .67149 Mdot
.24115 .08842 Mdot
.10611 .2177 Mdot
.34967 .0456 Mdot
.83519 .78539 Mdot
.87492 .74107 Mdot
.45102 .63372 Mdot
.36065 .35948 Mdot
.46151 .65032 Mdot
.36434 .34022 Mdot
.35812 .37773 Mdot
.43344 .60241 Mdot
.44164 .61771 Mdot
.38394 .2791 Mdot
.48562 .6845 Mdot
.49858 .70161 Mdot
.47313 .66732 Mdot
.36934 .32018 Mdot
.37581 .29969 Mdot
.41193 .54885 Mdot
.35511 .45269 Mdot
.35505 .43965 Mdot
.35527 .46501 Mdot
.40679 .52587 Mdot
.40896 .53714 Mdot
.42648 .58789 Mdot
.35564 .4108 Mdot
.35654 .39483 Mdot
.35518 .42571 Mdot
.41577 .56115 Mdot
.42062 .57415 Mdot
.49515 .14597 Mdot
.58836 .81034 Mdot
.5959 .81905 Mdot
.58014 .80051 Mdot
.47075 .1647 Mdot
.4834 .15467 Mdot
.52524 .12605 Mdot
.60876 .83379 Mdot
.6149 .83977 Mdot
.60256 .82687 Mdot
.50615 .13845 Mdot
.51624 .13189 Mdot
.41724 .22122 Mdot
.52513 .73449 Mdot
.5378 .74985 Mdot
.51185 .71837 Mdot
.3937 .25885 Mdot
.40492 .23942 Mdot
.45741 .17628 Mdot
.561 .77742 Mdot
.57112 .78952 Mdot
.54975 .76421 Mdot
.43027 .20454 Mdot
.44378 .18954 Mdot
.35582 .60359 Mdot
.433 .39622 Mdot
.35487 .58741 Mdot
.42594 .4109 Mdot
.44127 .38075 Mdot
.35995 .63933 Mdot
.35741 .62088 Mdot
.41125 .45035 Mdot
.35428 .55827 Mdot
.35436 .54512 Mdot
.35441 .57232 Mdot
.41997 .4248 Mdot
.41512 .43793 Mdot
.38377 .72075 Mdot
.49925 .2959 Mdot
.48602 .31318 Mdot
.51266 .27898 Mdot
.40511 .76078 Mdot
.39368 .74118 Mdot
.36369 .65884 Mdot
.46154 .34776 Mdot
.45086 .36455 Mdot
.47329 .33055 Mdot
.37551 .69993 Mdot
.36883 .67915 Mdot
.41226 .55184 Mdot
.35423 .44961 Mdot
.35419 .43616 Mdot
.35439 .46227 Mdot
.40663 .52828 Mdot
.40903 .53982 Mdot
.4279 .59206 Mdot
.35496 .4063 Mdot
.35605 .38974 Mdot
.35439 .42175 Mdot
.4164 .56449 Mdot
.42161 .57789 Mdot
.40475 .48461 Mdot
.35462 .52065 Mdot
.35471 .50902 Mdot
.35449 .53263 Mdot
.40826 .46221 Mdot
.40611 .47359 Mdot
.40506 .51715 Mdot
.35467 .48613 Mdot
.35455 .4744 Mdot
.35473 .49761 Mdot
.40412 .49546 Mdot
.40423 .50627 Mdot
.65598 .13623 Mdot
.55944 .90733 Mdot
.64812 .13825 Mdot
.55796 .90241 Mdot
.55951 .9126 Mdot
.67526 .13467 Mdot
.66494 .13499 Mdot
.54691 .88856 Mdot
.63407 .14397 Mdot
.628 .14789 Mdot
.64078 .14079 Mdot
.55553 .89765 Mdot
.55182 .89307 Mdot
.73273 .14607 Mdot
.52154 .94598 Mdot
.5368 .93844 Mdot
.49841 .95352 Mdot
.77765 .16812 Mdot
.75356 .155 Mdot
.68665 .13516 Mdot
.55381 .92451 Mdot
.55757 .91836 Mdot
.54733 .93119 Mdot
.71527 .14053 Mdot
.6999 .13699 Mdot
.45844 .82407 Mdot
.57245 .20753 Mdot
.56215 .21958 Mdot
.58154 .19659 Mdot
.48472 .84553 Mdot
.47187 .83558 Mdot
.41773 .7791 Mdot
.53899 .24731 Mdot
.52605 .26271 Mdot
.55102 .23288 Mdot
.44476 .81084 Mdot
.43111 .79584 Mdot
.5265 .87382 Mdot
.61599 .15752 Mdot
.61008 .16353 Mdot
.62206 .1524 Mdot
.54108 .88393 Mdot
.53442 .87904 Mdot
.49665 .85412 Mdot
.59706 .17805 Mdot
.58963 .18675 Mdot
.60392 .17038 Mdot
.51746 .86807 Mdot
.50749 .86157 Mdot
.63513 .85437 Mdot
.54783 .11006 Mdot
.64185 .85731 Mdot
.55215 .10559 Mdot
.54273 .11461 Mdot
.62296 .8467 Mdot
.629 .85079 Mdot
.55812 .09123 Mdot
.65753 .86128 Mdot
.6667 .86227 Mdot
.64943 .85953 Mdot
.55538 .10103 Mdot
.55723 .09623 Mdot
.59904 .82318 Mdot
.50216 .14248 Mdot
.5122 .13569 Mdot
.49103 .15024 Mdot
.58491 .80617 Mdot
.59229 .81517 Mdot
.6171 .84196 Mdot
.52941 .12427 Mdot
.53667 .11932 Mdot
.5212 .12966 Mdot
.60551 .83017 Mdot
.61144 .83642 Mdot
.39451 .04058 Mdot
.81328 .80214 Mdot
.84905 .76824 Mdot
.7831 .82446 Mdot
.48867 .04515 Mdot
.45127 .03983 Mdot
.06151 .30825 Mdot
.9321 .63365 Mdot
.956 .52211 Mdot
.89051 .7153 Mdot
.3088 .05947 Mdot
.18722 .1291 Mdot
.55043 .0738 Mdot
.68913 .86143 Mdot
.70276 .8592 Mdot
.67736 .86227 Mdot
.55762 .08589 Mdot
.55501 .08005 Mdot
.51422 .05234 Mdot
.73665 .84887 Mdot
.75817 .83896 Mdot
.71862 .85511 Mdot
.54298 .06705 Mdot
.53119 .05979 Mdot
.35604 .42747 Mdot
.41584 .55973 Mdot
.35641 .41296 Mdot
.42047 .57244 Mdot
.41214 .54766 Mdot
.35598 .45387 Mdot
.35592 .44108 Mdot
.44062 .615 Mdot
.35873 .38071 Mdot
.36107 .36294 Mdot
.35723 .3974 Mdot
.42606 .58586 Mdot
.43276 .60005 Mdot
.35643 .49996 Mdot
.40531 .49342 Mdot
.40527 .50388 Mdot
.40604 .48286 Mdot
.3563 .52232 Mdot
.3564 .51105 Mdot
.35613 .46597 Mdot
.40723 .52513 Mdot
.4093 .53618 Mdot
.4059 .51439 Mdot
.35639 .48883 Mdot
.35628 .47754 Mdot
.54604 .75953 Mdot
.42703 .21009 Mdot
.44027 .1949 Mdot
.41434 .22689 Mdot
.52145 .72989 Mdot
.53397 .7452 Mdot
.5765 .79621 Mdot
.46656 .16952 Mdot
.47908 .15919 Mdot
.45359 .18138 Mdot
.55716 .77286 Mdot
.56724 .78512 Mdot
.47091 .66353 Mdot
.36904 .32468 Mdot
.37515 .30459 Mdot
.36444 .3442 Mdot
.44957 .63064 Mdot
.45969 .64688 Mdot
.50857 .71384 Mdot
.39206 .26434 Mdot
.40262 .24507 Mdot
.38284 .28433 Mdot
.48288 .68042 Mdot
.49551 .69729 Mdot
.38438 .71971 Mdot
.49887 .29672 Mdot
.3763 .69927 Mdot
.48596 .31372 Mdot
.51199 .28007 Mdot
.40526 .75905 Mdot
.3941 .73979 Mdot
.45148 .36426 Mdot
.36468 .65892 Mdot
.36095 .63972 Mdot
.36976 .67888 Mdot
.47349 .3308 Mdot
.46196 .34773 Mdot
.45731 .82159 Mdot
.5707 .20954 Mdot
.56059 .2215 Mdot
.57963 .19865 Mdot
.4831 .843 Mdot
.47051 .83306 Mdot
.41755 .77709 Mdot
.53775 .24887 Mdot
.52515 .26407 Mdot
.54955 .23461 Mdot
.44386 .80846 Mdot
.43054 .79361 Mdot
.40547 .48295 Mdot
.35545 .52235 Mdot
.35555 .51086 Mdot
.35532 .53421 Mdot
.40914 .46078 Mdot
.40692 .47205 Mdot
.40539 .51497 Mdot
.35554 .48833 Mdot
.35543 .47684 Mdot
.35558 .49961 Mdot
.40474 .49364 Mdot
.40471 .50429 Mdot
.42686 .41001 Mdot
.35678 .60445 Mdot
.3558 .58846 Mdot
.35839 .62152 Mdot
.44204 .38023 Mdot
.43386 .39551 Mdot
.41217 .44905 Mdot
.35514 .5596 Mdot
.3552 .54657 Mdot
.35531 .57353 Mdot
.42093 .42375 Mdot
.41606 .43675 Mdot
.53929 .93444 Mdot
.7252 .14517 Mdot
.70866 .14038 Mdot
.74464 .15281 Mdot
.50587 .94922 Mdot
.52598 .94175 Mdot
.55706 .91514 Mdot
.68167 .13606 Mdot
.67085 .13588 Mdot
.6944 .13759 Mdot
.54825 .92751 Mdot
.55405 .92104 Mdot
.26712 .92177 Mdot
.90684 .31 Mdot
.86413 .24732 Mdot
.94447 .40384 Mdot
.03382 .59001 Mdot
.13617 .81804 Mdot
.47668 .95593 Mdot
.79407 .18135 Mdot
.76708 .16407 Mdot
.82605 .20725 Mdot
.36629 .95424 Mdot
.43272 .95971 Mdot
.52448 .87137 Mdot
.61384 .15968 Mdot
.60788 .16565 Mdot
.61974 .15449 Mdot
.53902 .88149 Mdot
.53223 .87663 Mdot
.49474 .85161 Mdot
.59515 .18025 Mdot
.58767 .18888 Mdot
.6018 .17253 Mdot
.51552 .86559 Mdot
.50554 .85908 Mdot
.55642 .89969 Mdot
.65248 .13785 Mdot
.64493 .13999 Mdot
.66125 .13654 Mdot
.55838 .90963 Mdot
.5582 .90447 Mdot
.545 .88607 Mdot
.63143 .14593 Mdot
.62545 .14987 Mdot
.63805 .14272 Mdot
.55361 .89509 Mdot
.54994 .89054 Mdot
.85948 .23186 Mdot
.29759 .94182 Mdot
.82184 .19536 Mdot
.39092 .96411 Mdot
.16618 .85886 Mdot
.94493 .37909 Mdot
.90302 .28973 Mdot
.51831 .95175 Mdot
.76394 .15621 Mdot
.74153 .1459 Mdot
.79013 .17157 Mdot
.45203 .9652 Mdot
.49174 .95946 Mdot
.86002 .76586 Mdot
.38955 .03472 Mdot
.29565 .05757 Mdot
.45105 .03347 Mdot
.79046 .82649 Mdot
.82226 .8026 Mdot
.96429 .49822 Mdot
.04076 .35255 Mdot
.04206 .65059 Mdot
.16383 .142 Mdot
.90367 .70758 Mdot
.94544 .61772 Mdot
.55406 .89645 Mdot
.64401 .13813 Mdot
.63707 .14138 Mdot
.65169 .1356 Mdot
.56037 .90606 Mdot
.55793 .90114 Mdot
.53592 .88207 Mdot
.6246 .15003 Mdot
.61832 .15534 Mdot
.63076 .14539 Mdot
.54897 .89182 Mdot
.54291 .88708 Mdot
.55505 .92916 Mdot
.70631 .13524 Mdot
.6923 .13307 Mdot
.72278 .13944 Mdot
.53583 .94384 Mdot
.54774 .93619 Mdot
.56175 .91118 Mdot
.66929 .13245 Mdot
.65993 .13362 Mdot
.6802 .13231 Mdot
.55945 .92266 Mdot
.56168 .91665 Mdot
.50812 .13475 Mdot
.59851 .82191 Mdot
.51843 .12806 Mdot
.60563 .82981 Mdot
.5908 .81291 Mdot
.48442 .15132 Mdot
.49685 .14244 Mdot
.62436 .84831 Mdot
.53602 .11683 Mdot
.54295 .11183 Mdot
.52778 .12216 Mdot
.61204 .83686 Mdot
.61813 .84304 Mdot
.42894 .20302 Mdot
.53796 .75019 Mdot
.55051 .76513 Mdot
.52454 .73426 Mdot
.40242 .2395 Mdot
.41525 .22046 Mdot
.47109 .16164 Mdot
.57288 .7914 Mdot
.58238 .80273 Mdot
.56214 .77891 Mdot
.44301 .18739 Mdot
.45716 .17361 Mdot
.70632 .863 Mdot
.55463 .06954 Mdot
.54729 .06249 Mdot
.55914 .07609 Mdot
.68013 .86594 Mdot
.69232 .86515 Mdot
.76412 .84197 Mdot
.51769 .04687 Mdot
.49104 .03916 Mdot
.53535 .05482 Mdot
.72287 .85878 Mdot
.74169 .85229 Mdot
.64379 .86014 Mdot
.55387 .10242 Mdot
.55773 .09771 Mdot
.54886 .10706 Mdot
.63058 .85289 Mdot
.6369 .85687 Mdot
.66919 .8658 Mdot
.56143 .08761 Mdot
.56134 .08211 Mdot
.56013 .09277 Mdot
.65154 .86264 Mdot
.65983 .8646 Mdot
.35309 .4078 Mdot
.41992 .57634 Mdot
.35415 .3908 Mdot
.42629 .59085 Mdot
.4147 .56266 Mdot
.35248 .43824 Mdot
.35262 .42355 Mdot
.45321 .63943 Mdot
.35896 .35301 Mdot
.36319 .33247 Mdot
.35602 .37251 Mdot
.43397 .60624 Mdot
.44295 .62246 Mdot
.35301 .4893 Mdot
.403 .50306 Mdot
.40364 .51425 Mdot
.40313 .49195 Mdot
.353 .51302 Mdot
.35305 .50114 Mdot
.35255 .452 Mdot
.40736 .53743 Mdot
.41053 .54971 Mdot
.40509 .52567 Mdot
.35288 .47733 Mdot
.35271 .46498 Mdot
.5701 .78801 Mdot
.45204 .17697 Mdot
.46647 .16416 Mdot
.43746 .19168 Mdot
.54641 .76042 Mdot
.55876 .77485 Mdot
.59731 .82021 Mdot
.49328 .14364 Mdot
.50515 .13548 Mdot
.48033 .15312 Mdot
.58014 .79999 Mdot
.5891 .81073 Mdot
.49105 .69301 Mdot
.37649 .28941 Mdot
.38588 .26773 Mdot
.36895 .31116 Mdot
.46479 .65701 Mdot
.47755 .67495 Mdot
.53325 .7448 Mdot
.40958 .22672 Mdot
.4232 .2083 Mdot
.397 .24667 Mdot
.50509 .71088 Mdot
.51938 .72821 Mdot
.42717 .79444 Mdot
.54956 .23446 Mdot
.53676 .24966 Mdot
.5615 .22046 Mdot
.4558 .82447 Mdot
.44139 .8104 Mdot
.38906 .73658 Mdot
.50901 .2829 Mdot
.49501 .30055 Mdot
.52315 .26588 Mdot
.41341 .77664 Mdot
.40056 .75723 Mdot
.50755 .86401 Mdot
.60571 .16872 Mdot
.59858 .17677 Mdot
.61215 .16157 Mdot
.5276 .8767 Mdot
.51809 .87076 Mdot
.46997 .83669 Mdot
.58208 .1962 Mdot
.57244 .20775 Mdot
.59067 .18587 Mdot
.49609 .85623 Mdot
.48351 .8472 Mdot
.3526 .56476 Mdot
.41627 .43173 Mdot
.41177 .44488 Mdot
.42183 .41783 Mdot
.35349 .59582 Mdot
.35286 .57974 Mdot
.35288 .52509 Mdot
.40575 .46921 Mdot
.40403 .48072 Mdot
.4083 .45732 Mdot
.35259 .55075 Mdot
.35271 .53758 Mdot
.36496 .67218 Mdot
.46845 .33639 Mdot
.45657 .35402 Mdot
.48141 .31848 Mdot
.37922 .71517 Mdot
.3712 .69354 Mdot
.35478 .61313 Mdot
.43662 .38751 Mdot
.42856 .4031 Mdot
.44597 .37111 Mdot
.36028 .65147 Mdot
.35696 .63171 Mdot
.44028 .03498 Mdot
.79417 .82169 Mdot
.37489 .03873 Mdot
.82661 .79614 Mdot
.76734 .83822 Mdot
.51185 .04702 Mdot
.48313 .0397 Mdot
.94812 .60014 Mdot
.14169 .16802 Mdot
.03214 .39704 Mdot
.27555 .06824 Mdot
.8653 .75648 Mdot
.90892 .69422 Mdot
.55695 .07604 Mdot
.68164 .86467 Mdot
.69399 .86364 Mdot
.67068 .86468 Mdot
.56051 .08769 Mdot
.55977 .08212 Mdot
.53079 .05481 Mdot
.72514 .85647 Mdot
.74442 .84935 Mdot
.70857 .86094 Mdot
.55203 .06952 Mdot
.54368 .06241 Mdot
.71973 .1401 Mdot
.53588 .9415 Mdot
.54693 .9341 Mdot
.51876 .94931 Mdot
.75972 .156 Mdot
.73833 .14648 Mdot
.6781 .13337 Mdot
.55855 .92079 Mdot
.56038 .91497 Mdot
.55413 .92719 Mdot
.7039 .13627 Mdot
.69029 .13426 Mdot
.89493 .28138 Mdot
.18619 .87399 Mdot
.3112 .94448 Mdot
.05671 .68842 Mdot
.96127 .47961 Mdot
.93735 .36478 Mdot
.78546 .17083 Mdot
.45594 .96276 Mdot
.49383 .95686 Mdot
.39866 .96274 Mdot
.8524 .22733 Mdot
.81577 .1929 Mdot
.53071 .74141 Mdot
.40909 .23104 Mdot
.54343 .75663 Mdot
.42211 .21308 Mdot
.39713 .2504 Mdot
.50371 .70846 Mdot
.51726 .72531 Mdot
.46333 .16953 Mdot
.56629 .78376 Mdot
.57627 .79552 Mdot
.55528 .7708 Mdot
.43572 .19678 Mdot
.44953 .18226 Mdot
.45404 .63929 Mdot
.36079 .35312 Mdot
.36495 .33323 Mdot
.35789 .37201 Mdot
.43534 .60704 Mdot
.44407 .62282 Mdot
.49032 .69115 Mdot
.37776 .29166 Mdot
.38661 .27077 Mdot
.37058 .31261 Mdot
.46511 .65632 Mdot
.47728 .67369 Mdot
.54972 .10708 Mdot
.63131 .85313 Mdot
.63769 .85682 Mdot
.62528 .84878 Mdot
.53776 .11655 Mdot
.54415 .11171 Mdot
.55954 .09284 Mdot
.65249 .8621 Mdot
.6609 .86386 Mdot
.64486 .85973 Mdot
.55426 .10249 Mdot
.55745 .09775 Mdot
.50106 .14071 Mdot
.59298 .8157 Mdot
.60036 .82406 Mdot
.58505 .80618 Mdot
.47674 .15848 Mdot
.48942 .14895 Mdot
.53023 .1217 Mdot
.61309 .83814 Mdot
.61913 .84386 Mdot
.60698 .83152 Mdot
.51171 .13355 Mdot
.52149 .12729 Mdot
.40394 .48768 Mdot
.3538 .51743 Mdot
.40352 .49868 Mdot
.35388 .50574 Mdot
.35367 .52948 Mdot
.40709 .46511 Mdot
.40512 .47654 Mdot
.35365 .47036 Mdot
.40493 .52076 Mdot
.40681 .5322 Mdot
.40385 .50965 Mdot
.35388 .49413 Mdot
.35379 .48238 Mdot
.42418 .41362 Mdot
.3547 .60058 Mdot
.35385 .58435 Mdot
.35622 .61799 Mdot
.4394 .38328 Mdot
.4312 .39886 Mdot
.40991 .4532 Mdot
.35344 .55525 Mdot
.35353 .54204 Mdot
.35349 .56928 Mdot
.41836 .42757 Mdot
.41366 .44072 Mdot
.36647 .32343 Mdot
.45861 .64713 Mdot
.47043 .66462 Mdot
.44795 .63012 Mdot
.35811 .36373 Mdot
.36159 .34404 Mdot
.3909 .25979 Mdot
.49686 .70007 Mdot
.51066 .71749 Mdot
.48336 .68236 Mdot
.37291 .30224 Mdot
.38101 .28088 Mdot
.35337 .43075 Mdot
.41314 .55647 Mdot
.41777 .5696 Mdot
.40953 .54405 Mdot
.35347 .45791 Mdot
.35334 .44477 Mdot
.35581 .38228 Mdot
.43035 .59822 Mdot
.43848 .61378 Mdot
.4235 .58351 Mdot
.35369 .41572 Mdot
.35444 .39959 Mdot
.61157 .16225 Mdot
.52736 .87559 Mdot
.51838 .86974 Mdot
.53545 .88085 Mdot
.62348 .15087 Mdot
.6176 .15618 Mdot
.59073 .18586 Mdot
.49695 .85562 Mdot
.48465 .84689 Mdot
.50827 .86315 Mdot
.60501 .16908 Mdot
.59816 .17693 Mdot
.65013 .13671 Mdot
.55949 .90444 Mdot
.55683 .89966 Mdot
.56057 .90955 Mdot
.6677 .13371 Mdot
.65857 .13492 Mdot
.62966 .14638 Mdot
.54832 .89042 Mdot
.54251 .88572 Mdot
.553 .89505 Mdot
.64275 .13929 Mdot
.63609 .14257 Mdot
.48435 .31486 Mdot
.38205 .71915 Mdot
.37374 .69789 Mdot
.392 .74003 Mdot
.51168 .28003 Mdot
.49791 .29728 Mdot
.44889 .36694 Mdot
.36213 .6562 Mdot
.3586 .63656 Mdot
.36712 .67676 Mdot
.47157 .33252 Mdot
.45971 .34996 Mdot
.56247 .21933 Mdot
.45801 .82496 Mdot
.44396 .81142 Mdot
.4716 .83674 Mdot
.58233 .19583 Mdot
.57283 .20693 Mdot
.52527 .26338 Mdot
.41618 .77885 Mdot
.40343 .76007 Mdot
.42985 .79603 Mdot
.55106 .23293 Mdot
.53851 .24762 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[31]:=",
  ImageSize->{288, 288},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181000`40n007h00Ooooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00
oooo8Ooo000Wool20004ool00`00ooooo`02ool4000Tool20004ool00`00ooooo`02ool4000Wool2
000[ool20004ool00`00ooooo`02ool4000Sool20004ool00`00ooooo`02ool4000Rool002Koo`04
003ooooo0008ool00`00ooooo`0Tool01000ooooo`002_oo00<00?ooool09Ooo00@00?ooool002Wo
o`04003ooooo000:ool00`00ooooo`0Qool01000ooooo`002?oo00<00?ooool08ooo000Vool01000
ooooo`002Ooo00<00?ooool08ooo00@00?ooool000[oo`03003ooooo02Goo`04003ooooo000Yool0
1000ooooo`002_oo00<00?ooool08Ooo00@00?ooool000Woo`03003ooooo02;oo`008?oo10000_oo
00@00?ooool000[oo`03003ooooo01coo`@000;oo`04003ooooo000:ool00`00ooooo`0Uool01000
ooooo`00:Ooo00@00?ooool000[oo`03003ooooo027oo`04003ooooo000:ool00`00ooooo`0Qool0
02Koo`04003ooooo000;ool00`00ooooo`0Qool01000ooooo`002_oo00<00?ooool09Ooo00@00?oo
ool002Woo`04003ooooo000:ool00`00ooooo`0Qool01000ooooo`002ooo00<00?ooool08?oo000V
ool01000ooooo`002?oo00@00?ooool002?oo`04003ooooo0008ool3000Wool01000ooooo`00:Ooo
00@00?ooool000Soo`<002?oo`04003ooooo0008ool01000ooooo`008_oo000Wool2000:ool2000U
ool2000;ool00`00ooooo`0Vool2000[ool2000;ool00`00ooooo`0Rool2000:ool2000Sool00?oo
ob7oo`00oooo8Ooo003ooolQool001_oool000D0007oo`006ooo00<00?ooool00ooo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool02?oo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool02?oo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool0
2?oo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool02?oo00<00?ooool01ooo00<00?ooool01_oo0@000Ooo000K
ool00`00ooooo`0=ool00`00ooooo`0`ool00`00ooooo`0`ool00`00ooooo`0`ool00`00ooooo`0_
ool00`00ooooo`0Uool10001ool001_oo`03003ooooo0?ooo`7oo`40007oo`006ooo00<00?ooool0
oooo0Ooo0@000Ooo000Kool00`00ooooo`3oool1ool10001ool001_oo`8006Coo`8000koo`8008[o
o`80007oo`006ooo00<00?ooool0I?oo0P003_oo0P00R_oo0@000Ooo000Kool00`00ooooo`1Sool3
000;ool50008ool20020ool10001ool001_oo`03003ooooo067oo`800003ool00?oo00Coo`8000?o
o`800003ool00?oo008000Soo`80083oo`40007oo`006ooo00<00?ooool0G_oo0P000_oo0`001_oo
00<00?ooool00_oo0`000ooo0P000_oo1P001Ooo0P00NOoo0@000Ooo000Kool00`00ooooo`1Oool0
0`00ooooo`02ool00`00ooooo`02ool200000ooo00000004ool00`00ooooo`030003ool01000ool0
00001Ooo0`00NOoo0@000Ooo000Kool00`00ooooo`18ool60009ool20003ool2000;ool200000ooo
00000008ool30002ool20002ool80003ool2001dool10001ool001_oo`03003ooooo04Woo`03003o
o`00008000Woo`8000?oo`8000Coo`8000Koo`05003ooooo003oo`020004ool200000ooo003oo`02
ool00`00ool0000200000ooo000000020003ool3001dool10001ool001_oo`03003ooooo04goo`03
003ooooo00Soo`8000?oo`03003ooooo00?oo`@000Coo`8000?oo`@000;oo`03003ooooo008000;o
o`@000;oo`P000;oo`80077oo`40007oo`006ooo00<00?ooool0BOoo0P0000?oo`0000002ooo00<0
0?ooool02Ooo0P001Ooo00<00?ooool00_oo00@00?oo003oo`8000;oo`<000;oo`<000;oo`P00004
ool00000001aool10001ool001_oo`8004_oo`04003ooooo000>ool2000Hool01000oooooooo1000
00?oo`0000001@0000?oo`0000001`00Kooo0P000Ooo000Kool00`00ooooo`1Cool20008ool00`00
ooooo`0Mool00`00ool0000200001Ooo00000000ool02@00L?oo0@000Ooo000Kool00`00ooooo`14
ool20007ool20002ool200000ooo003oo`0Zool900000ooo00000005001_ool10001ool001_oo`03
003ooooo04Goo`03003ooooo00Koo`05003oooooool0000_ool01@00ool00000ool02P00Kooo0@00
0Ooo000Kool00`00ooooo`13ool3000oool=001^ool10001ool001_oo`03003ooooo04Coo`80043o
o``006koo`40007oo`006ooo00<00?ooool0Aooo0P00?ooo2@00Kooo0@000Ooo000Kool00`00oooo
o`0kool2000;ool00`00ooooo`0nool8001_ool10001ool000Soo`8000Coo`03003ooooo00;oo`@0
00Coo`03003ooooo03_oo`800003ool0000004[oo`L006ooo`40007oo`001ooo00@00?ooool000So
o`03003ooooo00Goo`03003ooooo03coo`04003ooooo001;ool5001`ool10001ool000Ooo`04003o
oooo0009ool00`00ooooo`04ool2002<ool4001`ool20001ool00003ool00000008000;oo`04003o
oooo000:ool00`00ooooo`03ool00`00ooooo`2;ool4001aool10001ool000Ooo`04003ooooo000;
ool00`00ooooo`02ool00`00ooooo`29ool5001bool10001ool000Ooo`04003ooooo0008ool01000
ooooo`001?oo00<00?ooool0>Ooo0P00Booo0P0000?oo`0000000P00Looo0@000Ooo0008ool2000:
ool20005ool00`00ooooo`0jool00`00ooooo`1:ool6001cool10001ool001_oo`03003ooooo03Oo
o`8004_oo`P007Coo`40007oo`006ooo00<00?ooool0>?oo00<00?ooool0B_oo1P00MOoo0@000Ooo
000Kool00`00ooooo`0Xool400000ooo0000001Bool9000Nool400001?oo00000?oo0P0000Coo`00
000004[oo`40007oo`006ooo00<00?ooool0:Ooo0`000_oo00<00?ooool00P00C_oo20007?oo2P00
00?oo`0000000_oo0P000_oo0P00A_oo0@000Ooo000Kool00`00ooooo`0Vool2000:ool00`00oooo
o`18ool;000KoolC0002ool20003ool20011ool10001ool001_oo`03003ooooo02Ooo`03003ooooo
05Coo`T001_ooaP000?oo`80047oo`40007oo`006ooo0P009?oo0P000_oo0P00D_oo2P006ooo6`00
0_oo0P00@?oo0P000Ooo000Kool00`00ooooo`0Tool01@00oooooooo0000DOoo2P006ooo5P000_oo
0P0000?oo`0000000`000ooo0`00>ooo0@000Ooo000Kool00`00ooooo`1jool8000Lool90002ool0
0`00ool00002ool600000ooo000000020002ool50003ool2000kool10001ool001_oo`03003ooooo
07Ooo`X001coo`L000goo`03003oo`0000@00003ool0000000<00003ool0000000@003_oo`40007o
o`006ooo00<00?ooool0:ooo0P00BOoo2P007?oo10005Ooo0P0000?oo`00000010000_oo0`0000?o
o`00ool00ooo0P00=Ooo0@000Ooo000Kool00`00ooooo`0/ool00`00ooooo`16ool700000ooo003o
o`0Lool4000Jool00`00ool00002ool200000ooo000000020005ool2000dool10001ool001_oo`03
003ooooo027oo`80053oo`T001koo`@001ooo`P00003ool0000000;oo`8003Coo`40007oo`006ooo
00<00?ooool08_oo00@00?oo000004coo`T001koo`<002;oo`@00003ool00?oo00;oo`D003Coo`40
007oo`006ooo00<00?ooool08_oo0P0000?oo`00ool0C?oo1`007_oo10009?oo00<00?ooool00ooo
0P0000Coo`00000003Coo`40007oo`006ooo00<00?ooool08ooo00<00?ooool0Booo20007_oo1000
:Ooo0P0000Coo`00000003Koo`40007oo`006ooo0P00LOoo1`007ooo1000:?oo0P0000?oo`000000
1@001Ooo0P00;?oo0P000Ooo000Kool00`00ooooo`1^ool9000Nool4000Zool01@00ool00000ool0
0P0000?oo`00ool01Ooo0P00;?oo0@000Ooo000Kool00`00ooooo`1_ool500000ooo003oo`0Nool3
000^ool00`00ooooo`05ool20002ool2000/ool10001ool001_oo`03003ooooo06koo`H001ooo`@0
03Coo`800003ool0000000D002_oo`40007oo`006ooo00<00?ooool0K?oo20007ooo0`00<ooo0P00
00Koo`00ooooo`00ool4000[ool10001ool001_oo`03003ooooo01koo`8004_oo`H00003ool00?oo
01koo`<003Goo`03003ooooo00?oo`800003ool00?oo02_oo`40007oo`006ooo00<00?ooool07ooo
00<00?ooool0B?oo1`008?oo1000=ooo1P00;_oo0@000Ooo000Kool00`00ooooo`1Zool7000Oool3
000jool300000ooo003oo`02000[ool10001ool001_oo`03003ooooo01[oo`8004goo`L0023oo`<0
047oo`03003ooooo02Woo`40007oo`006ooo00<00?ooool06?oo0P0000?oo`00ool0Booo1P008Ooo
0`00BOoo0P008ooo0@000Ooo000Kool2000Jool00`00ooooo`1;ool7000Pool30016ool20003ool2
000Qool20001ool001_oo`03003ooooo06Koo`P001ooo`<0047oo`@000;oo`8000Coo`03003ooooo
023oo`40007oo`006ooo00<00?ooool0Iooo10008Ooo1000@_oo00<00?oo00000ooo00<00?oo0000
0`008_oo0@000Ooo000Kool00`00ooooo`0Iool2001:ool6000Pool4001<ool4000Qool10001ool0
01_oo`03003ooooo01[oo`03003ooooo04Ooo`L001ooo`@004Ooo`8000Goo`<0027oo`40007oo`00
6ooo00<00?ooool0Hooo10000_oo0P007_oo1000BOoo1@009Ooo0@000Ooo000Kool00`00ooooo`0H
ool2001:ool4000Qool3001<ool3000Uool10001ool001_oo`03003ooooo01Woo`03003ooooo04Ko
o`L001ooo`<004ooo`03003ooooo0080027oo`40007oo`006ooo00<00?ooool0H_oo1`007_oo0`00
E?oo00<00?ooool07ooo0@000Ooo000Kool00`00ooooo`1Sool5000Oool3001fool10001ool001_o
o`03003ooooo067oo`H001ooo`<007Ooo`40007oo`006ooo0P00HOoo1P007_oo1000Mooo0P000Ooo
000Kool00`00ooooo`1Oool7000Nool3001Cool200000ooo00000002ool2000Mool10001ool001_o
o`03003ooooo063oo`H001goo`<005Goo`04003ooooo0003ool00`00ooooo`02ool2000Gool10001
ool001_oo`03003ooooo05ooo`800004ool00000000Mool3001Lool20005ool00`00ooooo`0Eool1
0001ool001_oo`03003ooooo063oo`<001koo`@005goo`03003ooooo00?oo`8001Koo`40007oo`00
6ooo00<00?ooool0G_oo1P007?oo0`00H_oo0P000_oo00<00?ooool05?oo0@000Ooo000Kool00`00
ooooo`0Bool2001:ool6000Kool4001Kool30005ool01000ool000005_oo0@000Ooo000Kool00`00
ooooo`0Cool00`00ooooo`17ool7000Kool3001Mool200000ooo00000003ool3000Fool10001ool0
00Soo`8000Coo`03003ooooo00;oo`@000Coo`03003ooooo05koo`800003ool0000001_oo`<006;o
o`<000;oo`8001Koo`40007oo`001ooo00@00?ooool000[oo`03003ooooo00?oo`03003ooooo05go
o`D001coo`8006Goo`05003oooooool0000Fool10001ool000Ooo`04003ooooo000:ool00`00oooo
o`03ool2001Mool6000Kool3001Tool2000Iool20001ool00003ool00000008000;oo`04003ooooo
000:ool00`00ooooo`03ool00`00ooooo`1Mool5000Jool3001Vool00`00ooooo`0Hool10001ool0
00Ooo`04003ooooo000:ool00`00ooooo`03ool00`00ooooo`0<ool2001=ool20002ool2000Jool3
001Zool2000Fool10001ool000Ooo`04003ooooo0008ool30005ool00`00ooooo`0=ool00`00oooo
o`1<ool20002ool00`00ooooo`0Gool4001[ool00`00ooooo`0Dool10001ool000Soo`8000_oo`03
003ooooo00?oo`03003ooooo05_oo`H001Soo`D008;oo`40007oo`006ooo00<00?ooool0Fooo1000
00?oo`00ool05ooo0`00Q?oo0@000Ooo000Kool00`00ooooo`0?ool20019ool6000Hool30025ool1
0001ool001_oo`03003ooooo013oo`03003ooooo04Soo`D001Ooo`@006Woo`8001[oo`40007oo`00
6ooo00<00?ooool01ooo0`00D?oo1@006?oo1000J_oo00@00?oo000001Ooo`40007oo`006ooo00<0
0?ooool02?oo0P00Cooo1P005_oo1000Kooo00<00?ooool05Ooo0@000Ooo000Kool2001Kool5000G
ool2001dool2000Aool20001ool001_oo`03003ooooo00Koo`8005;oo`D001Goo`@007Goo`03003o
oooo013oo`40007oo`006ooo00<00?ooool01ooo00<00?ooool0Cooo1P005Ooo1000R?oo0@000Ooo
000Kool00`00ooooo`07ool2001@ool5000Fool3001eool2000Bool10001ool001_oo`03003ooooo
00Soo`03003ooooo04ooo`@001Coo`@007;oo`8000?oo`03003ooooo013oo`40007oo`006ooo00<0
0?ooool0F?oo1P005?oo1000Looo0P002?oo0P002ooo0@000Ooo000Kool00`00ooooo`04ool2001C
ool5000Cool4001eool00`00ooooo`07ool00`00ooooo`09ool10001ool001_oo`03003ooooo00Go
o`03003ooooo053oo`H001?oo`<007Soo`8000;oo`800003ool0000000_oo`40007oo`006ooo00<0
0?ooool02Ooo0P00C_oo0`0000?oo`00ool04Ooo1000NOoo00D00?ooooooo`0000;oo`03003ooooo
00Woo`40007oo`006ooo00<00?ooool02_oo00<00?ooool0Booo1@004_oo1@00S?oo0@000Ooo000K
ool00`00ooooo`1Hool5000Bool4001kool20002ool3000;ool10001ool001_oo`8000?oo`8005Go
o`@0017oo`@007goo`04003oooooool2000:ool20001ool001_oo`03003ooooo00?oo`04003ooooo
ool2001?ool5000Aool4001lool20003ool2000;ool10001ool001_oo`03003ooooo00Soo`03003o
oooo04goo`D0013oo`@007koo`03003ooooo00;oo`03003ooooo00Woo`40007oo`006ooo00<00?oo
ool0Eooo1P004?oo0`00T?oo0@000Ooo000Kool00`00ooooo`05ool2001@ool6000?ool4002@ool1
0001ool001_oo`03003ooooo00Koo`03003ooooo04ooo`D000ooo`@0093oo`40007oo`006ooo00<0
0?ooool0F?oo10003ooo1000P_oo0P003Ooo0@000Ooo000Kool00`00ooooo`1Gool5000?ool30024
ool00`00ooooo`0;ool10001ool001_oo`03003ooooo05Soo`D000goo`@007ooo`80017oo`40007o
o`006ooo00<00?ooool0Eooo100000?oo`00ool03?oo1000P?oo00<00?ooool03ooo0@000Ooo000K
ool20008ool2001>ool5000=ool4002Bool20001ool001_oo`03003ooooo00Soo`03003ooooo04go
o`D000coo`@009?oo`40007oo`006ooo00<00?ooool0Eooo1P002ooo1000Pooo0P003ooo0@000Ooo
000Kool00`00ooooo`1Gool5000<ool40024ool00`00ooooo`0=ool10001ool001_oo`03003ooooo
05Ooo`D000_oo`D009Coo`40007oo`006ooo00<00?ooool0Eooo1@003?oo0`00UOoo0@000Ooo000K
ool00`00ooooo`1Hool4000;ool4002Eool10001ool001_oo`03003ooooo05Ooo`D000_oo`@008Wo
o`8000[oo`40007oo`006ooo00<00?ooool0Eooo1@002_oo1000Rooo00<00?ooool02?oo0@000Ooo
000Kool00`00ooooo`1Gool5000:ool40025ool20002ool2000;ool10001ool001_oo`03003ooooo
00Coo`80057oo`D000[oo`@008Koo`8000;oo`03003ooooo00Woo`40007oo`006ooo0P001_oo0`00
Cooo1@002_oo1000Qooo00<00?ooool02ooo0P000Ooo000Kool00`00ooooo`07ool00`00ooooo`1>
ool40009ool4002Gool10001ool001_oo`03003ooooo05Ooo`D000Woo`@009Ooo`40007oo`006ooo
00<00?ooool00ooo0P000_oo0P00C_oo1@002Ooo1000ROoo0P003?oo0@000Ooo000Kool00`00oooo
o`04ool01@00oooooooo0000C_oo1@002Ooo1000R_oo00<00?ooool00P000_oo0P001?oo0@000Ooo
000Kool00`00ooooo`1Gool50009ool4002>ool01@00oooooooo00001?oo0@000Ooo000Kool00`00
ooooo`1Gool50009ool4002Gool10001ool001_oo`03003ooooo05Ooo`D000Woo`@008[oo`8000Co
o`8000Goo`40007oo`005?oo0P001Ooo00<00?ooool0Eooo1@002?oo1@00R_oo0P001Ooo00<00?oo
ool00ooo0@000Ooo000Cool01000ooooo`001?oo00<00?ooool0Eooo1@002?oo1@00Rooo00<00?oo
ool00P001ooo0@000Ooo000Cool01000ooooo`001?oo0P00F?oo1@002Ooo0`00T?oo0`001?oo0P00
0Ooo000Cool01000ooooo`001?oo00<00?ooool0Eooo1@002?oo1000Sooo10001Ooo0@000Ooo000C
ool01000ooooo`001?oo00<00?ooool0F?oo10002Ooo0`00T?oo0P001_oo0@000Ooo000Cool01000
ooooo`001?oo00<00?ooool0Eooo1@002Ooo1000Uooo0@000Ooo000Dool20005ool00`00ooooo`1H
ool40009ool4002Gool10001ool001_oo`03003ooooo05Ooo`D000Woo`@008Koo`8000?oo`8000[o
o`40007oo`006ooo00<00?ooool0Eooo1@002Ooo1000Qooo00<00?ooool00_oo00<00?ooool02?oo
0@000Ooo000Kool00`00ooooo`1Hool40009ool4002Gool10001ool001_oo`03003ooooo05Ooo`D0
00Woo`@009Ooo`40007oo`006ooo00<00?ooool0F?oo10002_oo0`00Uooo0@000Ooo000Kool20007
ool2001?ool5000:ool30028ool2000<ool20001ool001_oo`03003ooooo00Ooo`03003ooooo04go
o`D000[oo`@008Soo`03003ooooo00_oo`40007oo`006ooo00<00?ooool0F?oo10002_oo1000U_oo
0@000Ooo000Kool00`00ooooo`1Gool5000;ool4002Eool10001ool001_oo`03003ooooo05Soo`@0
00_oo`@009Goo`40007oo`006ooo00<00?ooool0Eooo1@002ooo1000UOoo0@000Ooo000Kool00`00
ooooo`1Hool4000<ool40022ool2000@ool10001ool001_oo`03003ooooo05Ooo`D000goo`<008?o
o`03003ooooo00koo`40007oo`006ooo00<00?ooool0F?oo10003?oo1000Pooo0P000ooo0P002_oo
0@000Ooo000Kool00`00ooooo`08ool2001=ool5000=ool40023ool01P00ool00000ool000[oo`40
007oo`006ooo00<00?ooool02Ooo00<00?ooool0C?oo10003Ooo1000Q_oo00<00?ooool02_oo0@00
0Ooo000Kool2001Hool5000>ool3002Bool20001ool001_oo`03003ooooo05Soo`@000koo`@0087o
o`8000ooo`40007oo`006ooo00<00?ooool01Ooo0P00D?oo1@003_oo1000P_oo00<00?ooool03Ooo
0@000Ooo000Kool00`00ooooo`06ool00`00ooooo`1?ool4000?ool3002Bool10001ool001_oo`03
003ooooo05Soo`@0013oo`<0097oo`40007oo`006ooo00<00?ooool0Eooo1P003ooo1000P?oo1000
3?oo0@000Ooo000Kool00`00ooooo`1Hool5000?ool4001oool300000ooo003oo`0;ool10001ool0
01_oo`03003ooooo05Soo`@0017oo`<0083oo`03003ooooo00?oo`8000Soo`40007oo`006ooo00<0
0?ooool0F?oo10004_oo0`00Q_oo00<00?ooool01_oo0@000Ooo000Kool00`00ooooo`1Hool5000A
ool40020ool4000:ool10001ool001_oo`8005Woo`<00003ool00?oo017oo`<0087oo`03003oo`00
00Woo`80007oo`006ooo00<00?ooool0F?oo1@004ooo0`00Mooo0P001_oo0`002ooo0@000Ooo000K
ool00`00ooooo`1Hool5000Cool3001hool00`00ooooo`05ool2000;ool10001ool001_oo`03003o
oooo05Woo`D001?oo`<007[oo`80013oo`40007oo`006ooo00<00?ooool01Ooo0`00DOoo1@004ooo
0`00Nooo00<00?ooool03_oo0@000Ooo000Kool00`00ooooo`06ool2001@ool6000Dool4001dool2
000Dool10001ool001_oo`03003ooooo00[oo`8004goo`D001Goo`<007Goo`03003ooooo01;oo`40
007oo`006ooo00<00?ooool02ooo00<00?ooool0Booo1@005Ooo1000ROoo0@000Ooo000Kool00`00
ooooo`07ool2001Aool4000Fool30029ool10001ool001_oo`03003ooooo00Koo`<0057oo`@001Oo
o`<008Soo`40007oo`006ooo0P002?oo00<00?ooool0D?oo10005ooo0`00Kooo0P005_oo0P000Ooo
000Kool00`00ooooo`0;ool2001<ool6000Gool3001^ool2000Gool10001ool001_oo`03003ooooo
00coo`03003ooooo04_oo`800003ool0000001Soo`<006koo`03003ooooo01Goo`40007oo`006ooo
00<00?ooool0F_oo1@006?oo1000K_oo10004ooo0@000Ooo000Kool00`00ooooo`1Kool3000Jool3
001_ool00`00ool0000Cool10001ool001_oo`03003ooooo00Ooo`8005;oo`D001Soo`@006[oo`80
01Soo`40007oo`006ooo00<00?ooool02?oo00<00?ooool0D?oo1@006Ooo1000J_oo00<00?ooool0
5_oo0@000Ooo000Kool00`00ooooo`0>ool2001<ool200000ooo0000000Iool30023ool10001ool0
01_oo`03003ooooo00_oo`8000;oo`03003ooooo04_oo`@001[oo`<006Soo`8001Soo`40007oo`00
2?oo0P001?oo00<00?ooool00_oo10001?oo00<00?ooool03?oo00<00?ooool0COoo1@006ooo0`00
I_oo1@005_oo0@000Ooo0007ool01000ooooo`002_oo00<00?ooool00ooo00<00?ooool0GOoo1000
7?oo0`00I_oo00@00?ooool001Koo`40007oo`001ooo00@00?ooool000[oo`03003ooooo00?oo`80
00[oo`8005;oo`D001_oo`<0063oo`8001goo`80007oo`001ooo00@00?ooool000[oo`03003ooooo
00?oo`03003ooooo00[oo`03003ooooo053oo`D001coo`80067oo`03003ooooo00Ooo`8001?oo`40
007oo`001ooo00@00?ooool000[oo`03003ooooo00?oo`03003ooooo05koo`<001koo`<006Goo`@0
0003ool00?oo01;oo`40007oo`001ooo00@00?ooool000Soo`<000Goo`03003ooooo01;oo`8004_o
o`@001goo`8006Goo`@001Goo`40007oo`002?oo0P002ooo00<00?ooool00ooo00<00?ooool04ooo
0P00BOoo1@007_oo0P00GOoo0P0000?oo`0000000_oo0`005_oo0@000Ooo000Kool00`00ooooo`0D
ool00`00ooooo`18ool5000Nool2001Mool01000ooooo`000ooo00<00?ooool05Ooo0@000Ooo000K
ool00`00ooooo`1Oool5000Nool3001Hool2000Qool10001ool001_oo`03003ooooo017oo`8004co
o`800003ool0000001ooo`<005Koo`<0027oo`40007oo`006ooo00<00?ooool04_oo00<00?ooool0
Booo0P0000?oo`0000007ooo0P00Eooo00<00?ooool08?oo0@000Ooo000Kool00`00ooooo`1Qool6
000Nool3001hool10001ool001_oo`8006;oo`H001ooo`8005Coo`D001koo`80007oo`006ooo00<0
0?ooool0H_oo10008?oo0`00E?oo00@00?oo000001ooo`40007oo`006ooo00<00?ooool04ooo0P00
COoo1P007ooo0`00M_oo0@000Ooo000Kool00`00ooooo`0Dool00`00ooooo`1;ool400000ooo0000
000Oool3001;ool20002ool3000Sool10001ool001_oo`03003ooooo06?oo`H0023oo`<004_oo`04
003oooooool4000Qool10001ool001_oo`03003ooooo06Coo`@002;oo`8005;oo`03003ooooo01oo
o`40007oo`006ooo00<00?ooool0IOoo1@008Ooo0P00BOoo0P00:?oo0@000Ooo000Kool00`00oooo
o`1Uool200000ooo00000002000Oool30015ool20002ool01@00ool00000ool010008?oo0@000Ooo
000Kool00`00ooooo`0Jool2001:ool6000Pool30014ool20005ool6000Pool10001ool001_oo`03
003ooooo01_oo`03003ooooo04Soo`D002;oo`8004Goo`03003ooooo00Goo`<0027oo`40007oo`00
6ooo0P00J?oo1P008Ooo0P00Kooo0P000Ooo000Kool00`00ooooo`0Nool20017ool7000Qool20011
ool4000Zool10001ool001_oo`03003ooooo01ooo`03003ooooo04Koo`H0027oo`<003coo`8000?o
o`04003oo`00000Yool10001ool001_oo`03003ooooo06[oo`@002;oo`<003coo`04003oo`000003
ool00`00ooooo`0Wool10001ool001_oo`03003ooooo06[oo`L0023oo`<003koo`03003ooooo02_o
o`40007oo`006ooo00<00?ooool0Jooo1P008Ooo0P00=ooo10001?oo0P00:ooo0@000Ooo000Kool0
0`00ooooo`1/ool00`00ooooo`02000Rool2000fool200001Ooo00000000ool00`00:ooo0@000Ooo
000Kool00`00ooooo`1]ool7000Oool3000fool01000oooooooo1000;?oo0@000Ooo000Kool00`00
ooooo`1^ool6000Pool3000_ool20002ool30005ool4000Zool10001ool001_oo`03003ooooo06ko
o`L0023oo`<002ooo`03003oo`0000@000Goo`<002[oo`40007oo`006ooo00<00?ooool08?oo0P00
COoo0P0000?oo`0000000`007ooo0`00:ooo10000_oo00@00?ooool003;oo`40007oo`006ooo0P00
8_oo00<00?ooool0C?oo1`008?oo0`00:_oo0P0000Coo`00ooooo`8003?oo`80007oo`006ooo00<0
0?ooool0L_oo20007_oo0`009Ooo0P0000Coo`00000000;oo`@003Coo`40007oo`006ooo00<00?oo
ool0L_oo0`0000?oo`0000000P007ooo0P008ooo0P0000?oo`0000000`000ooo0P00=Ooo0@000Ooo
000Kool00`00ooooo`1cool9000Nool3000Nool200001?oo000000000_oo10000_oo1000<ooo0@00
0Ooo000Kool00`00ooooo`1eool8000Nool3000Nool50002ool200000ooo003oo`02ool3000cool1
0001ool001_oo`03003ooooo02Ooo`8004goo`T001goo`<001Ooo`800003ool0000000D00003ool0
0000008003_oo`40007oo`006ooo00<00?ooool0:?oo00<00?ooool0C?oo20007_oo10005?oo0`00
0_oo0P0000?oo`0000000P0000Coo`00000003_oo`40007oo`006ooo00<00?ooool0N?oo2@007Ooo
1`002_oo1@0000?oo`0000000`000_oo1@000_oo0`00>_oo0@000Ooo000Kool00`00ooooo`0Yool2
001>ool:000Lool@00000ooo0000000:00001?oo000000001?oo0P00>_oo0@000Ooo000Kool00`00
ooooo`0Uool20003ool01000ool00000COoo2@007?oo700000?oo`000000@Ooo0@000Ooo000Kool2
000Wool30004ool00`00ooooo`1<ool300000ooo00000003000LoolF00001?oo000000000ooo0P00
?ooo0P000Ooo000Kool00`00ooooo`0Wool20007ool3001:ool:000LoolD00000ooo00000005ool0
0`00ooooo`0nool10001ool001_oo`03003ooooo02Soo`8000Ooo`8004_oo`X001goo`d000;oo`80
00?oo`8004Goo`40007oo`006ooo00<00?ooool0:Ooo00<00?ooool0EOoo20007_oo200000Goo`00
00000?oo008000Coo`03003ooooo04?oo`40007oo`006ooo00<00?ooool0;Ooo0P00Dooo2@007_oo
00H00?oo00000?oo0002ool20002ool00`00ooooo`18ool10001ool001_oo`03003ooooo02koo`03
003ooooo05?oo`L007Goo`40007oo`006ooo00<00?ooool0:ooo0P001Ooo0P00DOoo1`00M?oo0@00
0Ooo000Kool00`00ooooo`0/ool00`00ooooo`04ool00`00ooooo`1Aool6001cool10001ool001_o
o`03003ooooo03Woo`800003ool0000004[oo`D007?oo`40007oo`002?oo0P001?oo00<00?ooool0
0_oo10001?oo00<00?ooool0;_oo0P002_oo00@00?ooool004_oo`D007;oo`40007oo`001ooo00@0
0?ooool000Soo`03003ooooo00Goo`03003ooooo02ooo`03003ooooo00?oo`8005Coo`@0077oo`40
007oo`001ooo00@00?ooool000Woo`03003ooooo00Coo`8003Ooo`03003ooooo057oo`D0073oo`80
007oo`001ooo00@00?ooool000[oo`03003ooooo00?oo`03003ooooo08[oo`H0073oo`40007oo`00
1ooo00@00?ooool000_oo`03003ooooo00;oo`03003ooooo08[oo`L006ooo`40007oo`001ooo00@0
0?ooool000Soo`04003ooooo0004ool00`00ooooo`28ool9001_ool10001ool000Soo`8000[oo`80
00Goo`03003ooooo08Koo`/006ooo`40007oo`006ooo00<00?ooool0@_oo0P000_oo0P00?_oo3000
K_oo0@000Ooo000Kool00`00ooooo`13ool01@00oooooooo00000_oo0P0000?oo`000000=?oo0P00
00Coo`00003oo`T006koo`40007oo`006ooo00<00?ooool0B?oo0P0000Goo`00ooooo`0000Koo`80
02coo`h006ooo`40007oo`006ooo00<00?ooool0BOoo00<00?ooool02_oo00<00?ooool09_oo0P00
0ooo0`0000?oo`0000001`00Kooo0@000Ooo000Kool00`00ooooo`1Dool2000Tool20002ool50000
0ooo0000000700001?oo00000000Kooo0@000Ooo000Kool20019ool40004ool20003ool00`00oooo
o`09ool20003ool2000:ool20004ool200000ooo003oo`02ool200001?oo003ooooo1@0000?oo`00
00000P00L?oo0P000Ooo000Kool00`00ooooo`19ool300000ooo00000002ool00`00ooooo`0=ool2
0004ool00`00ooooo`04ool20003ool00`00ooooo`02ool200000ooo00000002ool;0002ool3001a
ool10001ool001_oo`03003ooooo04[oo`05003oooooool00004ool20008ool200001?oo00000000
0ooo0P001ooo0P0000?oo`0000001?oo0P000_oo00@00?ooooooo`800004ool000000002ool5001c
ool10001ool001_oo`03003ooooo04_oo`8000Ooo`03003ooooo00Ooo`@000Goo`03003ooooo00?o
o`@000;oo`03003ooooo00?oo`800003ool0000000800003ool00?oo008000Coo`@007?oo`40007o
o`006ooo00<00?ooool0C?oo00@00?oo000000_oo`8000?oo`03003ooooo00;oo`8000Ooo`<000;o
o`8000?oo`<00005ool00000ool00004ool5001gool10001ool001_oo`03003ooooo04ooo`03003o
oooo00[oo`03003ooooo00Ooo`<000Coo`800003ool00?oo00;oo`<000;oo`03003ooooo008000Oo
o`@007Ooo`40007oo`006ooo00<00?ooool0DOoo0P003ooo10000_oo00<00?ooool00ooo00@00?oo
ooooo`<00003ool00?oo00Goo`H007goo`40007oo`006ooo00<00?ooool0D_oo00<00?ooool03?oo
0`0000?oo`00ool02ooo0`002Ooo0`0000?oo`00ool0O?oo0@000Ooo000Kool00`00ooooo`1Rool4
000=ool60027ool10001ool001_oo`03003ooooo06Goo`D000_oo`800003ool00?oo08Koo`40007o
o`006ooo0P00J?oo00<00?oo0000UOoo0P000Ooo000Kool00`00ooooo`3oool1ool10001ool001_o
o`03003ooooo0?ooo`7oo`40007oo`006ooo00<00?ooool0oooo0Ooo0@000Ooo000Kool00`00oooo
o`0=ool00`00ooooo`0`ool00`00ooooo`0`ool00`00ooooo`0`ool00`00ooooo`0_ool00`00oooo
o`0Uool10001ool001_oo`03003ooooo00?oo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo
00Soo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo
00Ooo`03003ooooo00Ooo`03003ooooo00Soo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo
00Ooo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo00Soo`03003ooooo00Ooo`03003ooooo
00Ooo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo00Ooo`03003ooooo
00Soo`03003ooooo00Ooo`03003ooooo00Koo`40007oo`006oooo`001@000Ooo003ooolQool00?oo
ob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.286362, -0.288422, \
0.00197108, 0.00193011}}]
}, Open  ]],

Cell["\<\
Since the H\[EAcute]non-Heiles system is Hamiltonian, a symplectic \
method gives much better qualitative results for this example.\
\>", "Text"],

Cell[TextData[{
  "This integrates the H\[EAcute]non-Heiles system using a fourth order \
symplectic partitioned Runge Kutta method with fixed step size of 0.25.  The \
event action is to ",
  StyleBox["Sow", "MR"],
  " the values of ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_2\ and\ \(\(Y\_4\)\(.\)\)\)]]
}], "MathCaption"],

Cell[BoxData[
    \(\(sdata = \[IndentingNewLine]Reap[\[IndentingNewLine]\(NDSolve[
              eqns, {}, \ {T, \ 
                10000}, \[IndentingNewLine]Method \[Rule] {EventLocator, \
"\<Event\>" \[Rule] Y\_1[T], "\<EventAction\>" \[RuleDelayed] \ 
                    Sow[{Y\_2[T], \ 
                        Y\_4[T]}], "\<EventLocationMethod\>" -> \
"\<LinearInterpolation\>", \[IndentingNewLine]"\<Method\>" \[Rule] \
{"\<SymplecticPartitionedRungeKutta\>", "\<DifferenceOrder\>" \[Rule] 
                        4, "\<PositionVariables\>" \[Rule] {Y\_1[T], \ 
                          Y\_2[T]}}}, \[IndentingNewLine]StartingStepSize \
\[Rule] 0.25, \ 
              MaxSteps \[Rule] \[Infinity]];\)\[IndentingNewLine]];\)\)], \
"Input",
  CellLabel->"In[33]:="],

Cell[TextData[{
  "This plots the  Poincar\[EAcute]  section.  The collected data is found in \
the last part of the result of ",
  StyleBox["Reap", "MR"],
  " and the list of points is the first part of that."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(psdata = sdata[\([\(-1\), 1]\)];\)\), "\n", 
    \(\(ListPlot[psdata, Axes \[Rule] False, Frame \[Rule] True, 
        AspectRatio \[Rule] 1];\)\)}], "Input",
  CellLabel->"In[34]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.460855 1.92647 0.500003 1.98973 [
[.07556 -0.0125 -12 -9 ]
[.07556 -0.0125 12 0 ]
[.26821 -0.0125 -12 -9 ]
[.26821 -0.0125 12 0 ]
[.46086 -0.0125 -3 -9 ]
[.46086 -0.0125 3 0 ]
[.6535 -0.0125 -9 -9 ]
[.6535 -0.0125 9 0 ]
[.84615 -0.0125 -9 -9 ]
[.84615 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .10206 -24 -4.5 ]
[-0.0125 .10206 0 4.5 ]
[-0.0125 .30103 -24 -4.5 ]
[-0.0125 .30103 0 4.5 ]
[-0.0125 .5 -6 -4.5 ]
[-0.0125 .5 0 4.5 ]
[-0.0125 .69898 -18 -4.5 ]
[-0.0125 .69898 0 4.5 ]
[-0.0125 .89795 -18 -4.5 ]
[-0.0125 .89795 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.07556 0 m
.07556 .00625 L
s
[(-0.2)] .07556 -0.0125 0 1 Mshowa
.26821 0 m
.26821 .00625 L
s
[(-0.1)] .26821 -0.0125 0 1 Mshowa
.46086 0 m
.46086 .00625 L
s
[(0)] .46086 -0.0125 0 1 Mshowa
.6535 0 m
.6535 .00625 L
s
[(0.1)] .6535 -0.0125 0 1 Mshowa
.84615 0 m
.84615 .00625 L
s
[(0.2)] .84615 -0.0125 0 1 Mshowa
.125 Mabswid
.11409 0 m
.11409 .00375 L
s
.15262 0 m
.15262 .00375 L
s
.19115 0 m
.19115 .00375 L
s
.22968 0 m
.22968 .00375 L
s
.30674 0 m
.30674 .00375 L
s
.34527 0 m
.34527 .00375 L
s
.3838 0 m
.3838 .00375 L
s
.42233 0 m
.42233 .00375 L
s
.49938 0 m
.49938 .00375 L
s
.53791 0 m
.53791 .00375 L
s
.57644 0 m
.57644 .00375 L
s
.61497 0 m
.61497 .00375 L
s
.69203 0 m
.69203 .00375 L
s
.73056 0 m
.73056 .00375 L
s
.76909 0 m
.76909 .00375 L
s
.80762 0 m
.80762 .00375 L
s
.03703 0 m
.03703 .00375 L
s
.88468 0 m
.88468 .00375 L
s
.92321 0 m
.92321 .00375 L
s
.96174 0 m
.96174 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .10206 m
.00625 .10206 L
s
[(-0.2)] -0.0125 .10206 1 0 Mshowa
0 .30103 m
.00625 .30103 L
s
[(-0.1)] -0.0125 .30103 1 0 Mshowa
0 .5 m
.00625 .5 L
s
[(0)] -0.0125 .5 1 0 Mshowa
0 .69898 m
.00625 .69898 L
s
[(0.1)] -0.0125 .69898 1 0 Mshowa
0 .89795 m
.00625 .89795 L
s
[(0.2)] -0.0125 .89795 1 0 Mshowa
.125 Mabswid
0 .14185 m
.00375 .14185 L
s
0 .18165 m
.00375 .18165 L
s
0 .22144 m
.00375 .22144 L
s
0 .26124 m
.00375 .26124 L
s
0 .34082 m
.00375 .34082 L
s
0 .38062 m
.00375 .38062 L
s
0 .42041 m
.00375 .42041 L
s
0 .46021 m
.00375 .46021 L
s
0 .5398 m
.00375 .5398 L
s
0 .57959 m
.00375 .57959 L
s
0 .61939 m
.00375 .61939 L
s
0 .65918 m
.00375 .65918 L
s
0 .73877 m
.00375 .73877 L
s
0 .77857 m
.00375 .77857 L
s
0 .81836 m
.00375 .81836 L
s
0 .85815 m
.00375 .85815 L
s
0 .06226 m
.00375 .06226 L
s
0 .02247 m
.00375 .02247 L
s
0 .93774 m
.00375 .93774 L
s
0 .97754 m
.00375 .97754 L
s
.25 Mabswid
0 0 m
0 1 L
s
.07556 .99375 m
.07556 1 L
s
.26821 .99375 m
.26821 1 L
s
.46086 .99375 m
.46086 1 L
s
.6535 .99375 m
.6535 1 L
s
.84615 .99375 m
.84615 1 L
s
.125 Mabswid
.11409 .99625 m
.11409 1 L
s
.15262 .99625 m
.15262 1 L
s
.19115 .99625 m
.19115 1 L
s
.22968 .99625 m
.22968 1 L
s
.30674 .99625 m
.30674 1 L
s
.34527 .99625 m
.34527 1 L
s
.3838 .99625 m
.3838 1 L
s
.42233 .99625 m
.42233 1 L
s
.49938 .99625 m
.49938 1 L
s
.53791 .99625 m
.53791 1 L
s
.57644 .99625 m
.57644 1 L
s
.61497 .99625 m
.61497 1 L
s
.69203 .99625 m
.69203 1 L
s
.73056 .99625 m
.73056 1 L
s
.76909 .99625 m
.76909 1 L
s
.80762 .99625 m
.80762 1 L
s
.03703 .99625 m
.03703 1 L
s
.88468 .99625 m
.88468 1 L
s
.92321 .99625 m
.92321 1 L
s
.96174 .99625 m
.96174 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .10206 m
1 .10206 L
s
.99375 .30103 m
1 .30103 L
s
.99375 .5 m
1 .5 L
s
.99375 .69898 m
1 .69898 L
s
.99375 .89795 m
1 .89795 L
s
.125 Mabswid
.99625 .14185 m
1 .14185 L
s
.99625 .18165 m
1 .18165 L
s
.99625 .22144 m
1 .22144 L
s
.99625 .26124 m
1 .26124 L
s
.99625 .34082 m
1 .34082 L
s
.99625 .38062 m
1 .38062 L
s
.99625 .42041 m
1 .42041 L
s
.99625 .46021 m
1 .46021 L
s
.99625 .5398 m
1 .5398 L
s
.99625 .57959 m
1 .57959 L
s
.99625 .61939 m
1 .61939 L
s
.99625 .65918 m
1 .65918 L
s
.99625 .73877 m
1 .73877 L
s
.99625 .77857 m
1 .77857 L
s
.99625 .81836 m
1 .81836 L
s
.99625 .85815 m
1 .85815 L
s
.99625 .06226 m
1 .06226 L
s
.99625 .02247 m
1 .02247 L
s
.99625 .93774 m
1 .93774 L
s
.99625 .97754 m
1 .97754 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
.008 w
.73601 .86381 Mdot
.55763 .05509 Mdot
.75698 .85596 Mdot
.5448 .04667 Mdot
.56591 .06291 Mdot
.70332 .87092 Mdot
.7185 .86837 Mdot
.4452 .024 Mdot
.81097 .82518 Mdot
.84636 .79605 Mdot
.78145 .84398 Mdot
.52462 .0378 Mdot
.49356 .02937 Mdot
.66005 .86812 Mdot
.56682 .09289 Mdot
.57028 .08777 Mdot
.56206 .09785 Mdot
.6451 .86129 Mdot
.6522 .86512 Mdot
.69009 .87194 Mdot
.57211 .07637 Mdot
.57034 .06997 Mdot
.57208 .08229 Mdot
.66903 .87021 Mdot
.67889 .87156 Mdot
.49542 .97024 Mdot
.80914 .17348 Mdot
.78018 .15536 Mdot
.84447 .20221 Mdot
.37422 .97106 Mdot
.44848 .9759 Mdot
.55833 .94446 Mdot
.73513 .13597 Mdot
.71769 .13148 Mdot
.75563 .14341 Mdot
.526 .96173 Mdot
.54549 .95291 Mdot
.10297 .21597 Mdot
.93555 .67428 Mdot
.97112 .56254 Mdot
.88895 .74924 Mdot
.36922 .02983 Mdot
.25189 .0724 Mdot
.25917 .9316 Mdot
.93325 .32107 Mdot
.88665 .24785 Mdot
.96999 .43091 Mdot
.02381 .50858 Mdot
.11045 .79549 Mdot
.36616 .33389 Mdot
.45668 .63851 Mdot
.37226 .31089 Mdot
.46908 .65728 Mdot
.44578 .62047 Mdot
.35909 .37657 Mdot
.36188 .35589 Mdot
.51282 .71502 Mdot
.39057 .26394 Mdot
.40274 .24123 Mdot
.38037 .28737 Mdot
.48272 .67651 Mdot
.49741 .69589 Mdot
.35614 .44547 Mdot
.41317 .54417 Mdot
.4171 .55764 Mdot
.41028 .53135 Mdot
.3565 .4732 Mdot
.3563 .45971 Mdot
.35739 .39584 Mdot
.42866 .58714 Mdot
.43648 .60332 Mdot
.42221 .57193 Mdot
.35615 .43019 Mdot
.3565 .41369 Mdot
.61802 .83777 Mdot
.51936 .12639 Mdot
.53052 .11952 Mdot
.50704 .13437 Mdot
.60208 .81984 Mdot
.61052 .82938 Mdot
.63828 .85674 Mdot
.54905 .10801 Mdot
.5562 .10286 Mdot
.54045 .11348 Mdot
.62487 .84512 Mdot
.63149 .85144 Mdot
.5574 .76775 Mdot
.43155 .20036 Mdot
.44729 .18295 Mdot
.41654 .21988 Mdot
.52824 .73358 Mdot
.54319 .75125 Mdot
.59279 .80892 Mdot
.47872 .15478 Mdot
.49345 .14374 Mdot
.46319 .16778 Mdot
.57061 .78285 Mdot
.58239 .79658 Mdot
.4074 .49363 Mdot
.35666 .51311 Mdot
.40736 .50556 Mdot
.35671 .50041 Mdot
.35651 .5261 Mdot
.41015 .46933 Mdot
.40832 .48163 Mdot
.35633 .46137 Mdot
.41001 .52985 Mdot
.41275 .54259 Mdot
.40823 .51758 Mdot
.35666 .48775 Mdot
.35652 .47482 Mdot
.42826 .41369 Mdot
.35731 .60318 Mdot
.35645 .5854 Mdot
.35898 .6224 Mdot
.44528 .38045 Mdot
.43605 .39755 Mdot
.41299 .45651 Mdot
.35616 .55381 Mdot
.35632 .53959 Mdot
.35613 .56899 Mdot
.42191 .42885 Mdot
.41688 .44307 Mdot
.37925 .2902 Mdot
.48107 .6742 Mdot
.4957 .69357 Mdot
.46746 .65501 Mdot
.36564 .33653 Mdot
.37147 .31364 Mdot
.41481 .22234 Mdot
.52632 .73143 Mdot
.54154 .74917 Mdot
.5109 .71278 Mdot
.38915 .26673 Mdot
.4011 .2439 Mdot
.35641 .41574 Mdot
.42156 .57019 Mdot
.42779 .58527 Mdot
.41658 .55601 Mdot
.35617 .44717 Mdot
.35615 .43203 Mdot
.36151 .3584 Mdot
.44464 .61838 Mdot
.45531 .63632 Mdot
.43545 .60133 Mdot
.35721 .39809 Mdot
.3588 .37897 Mdot
.54856 .89174 Mdot
.63777 .14344 Mdot
.53986 .88625 Mdot
.63119 .1489 Mdot
.64474 .13893 Mdot
.5619 .90187 Mdot
.55595 .89686 Mdot
.60996 .17103 Mdot
.51889 .87321 Mdot
.50642 .86517 Mdot
.52996 .88016 Mdot
.62463 .1553 Mdot
.61766 .16265 Mdot
.57224 .92329 Mdot
.68957 .12811 Mdot
.67842 .12854 Mdot
.70243 .1289 Mdot
.56618 .93672 Mdot
.57038 .92972 Mdot
.56653 .90688 Mdot
.65964 .13203 Mdot
.65193 .13513 Mdot
.66837 .1298 Mdot
.5721 .9174 Mdot
.57003 .91199 Mdot
.49664 .30513 Mdot
.38994 .73485 Mdot
.37985 .71138 Mdot
.40205 .75761 Mdot
.52746 .26738 Mdot
.51192 .28595 Mdot
.45604 .36245 Mdot
.36592 .66496 Mdot
.36175 .64304 Mdot
.37187 .68789 Mdot
.48203 .32451 Mdot
.46839 .34372 Mdot
.58184 .20411 Mdot
.47784 .84461 Mdot
.46231 .83149 Mdot
.49265 .85574 Mdot
.60164 .18071 Mdot
.59238 .19173 Mdot
.5425 .24967 Mdot
.43085 .7987 Mdot
.41585 .77906 Mdot
.44648 .81621 Mdot
.56983 .21786 Mdot
.55669 .23309 Mdot
.50854 .13335 Mdot
.60325 .82099 Mdot
.52091 .12555 Mdot
.61133 .83049 Mdot
.59409 .81024 Mdot
.4804 .15336 Mdot
.49497 .14252 Mdot
.63245 .85204 Mdot
.54146 .11278 Mdot
.54978 .10734 Mdot
.53192 .11879 Mdot
.61872 .83877 Mdot
.6258 .84586 Mdot
.41828 .21746 Mdot
.52997 .73576 Mdot
.54504 .75325 Mdot
.51458 .71728 Mdot
.39182 .26121 Mdot
.40421 .23863 Mdot
.46511 .16612 Mdot
.57196 .78461 Mdot
.58359 .79817 Mdot
.55911 .76959 Mdot
.43351 .19816 Mdot
.44932 .18102 Mdot
.72031 .86804 Mdot
.56497 .06198 Mdot
.5566 .05416 Mdot
.56989 .06914 Mdot
.69177 .87178 Mdot
.70492 .87076 Mdot
.78479 .84206 Mdot
.52159 .03675 Mdot
.48883 .02848 Mdot
.54272 .04561 Mdot
.73853 .86292 Mdot
.75952 .85492 Mdot
.65325 .86544 Mdot
.56289 .09731 Mdot
.56735 .0923 Mdot
.55697 .10226 Mdot
.63896 .85739 Mdot
.64576 .86187 Mdot
.68008 .8717 Mdot
.57215 .08161 Mdot
.57222 .07569 Mdot
.57038 .08708 Mdot
.6612 .86835 Mdot
.66996 .87051 Mdot
.41222 .45962 Mdot
.35619 .55034 Mdot
.40963 .4723 Mdot
.35637 .53632 Mdot
.35611 .56526 Mdot
.4206 .43232 Mdot
.41585 .44634 Mdot
.35671 .49739 Mdot
.40731 .49647 Mdot
.40749 .50841 Mdot
.40802 .48451 Mdot
.35655 .52297 Mdot
.35668 .51007 Mdot
.45333 .36682 Mdot
.36477 .65963 Mdot
.36097 .638 Mdot
.37027 .68235 Mdot
.47868 .32911 Mdot
.46532 .34824 Mdot
.42662 .41739 Mdot
.35705 .59882 Mdot
.35633 .58137 Mdot
.3585 .61769 Mdot
.44294 .38461 Mdot
.43407 .40148 Mdot
.36235 .3533 Mdot
.44705 .62258 Mdot
.45807 .64071 Mdot
.43749 .60531 Mdot
.3575 .39367 Mdot
.35933 .3742 Mdot
.38143 .28458 Mdot
.48448 .67881 Mdot
.49929 .69817 Mdot
.47058 .65955 Mdot
.36686 .33115 Mdot
.37316 .30808 Mdot
.35628 .45807 Mdot
.41057 .53284 Mdot
.41356 .54572 Mdot
.40857 .52047 Mdot
.35663 .4847 Mdot
.35647 .47167 Mdot
.35655 .41166 Mdot
.42293 .5737 Mdot
.42948 .589 Mdot
.41766 .55931 Mdot
.35615 .44366 Mdot
.35619 .42826 Mdot
.52554 .26953 Mdot
.41401 .77657 Mdot
.51016 .28823 Mdot
.40042 .75494 Mdot
.42894 .79645 Mdot
.55516 .23504 Mdot
.54066 .2517 Mdot
.37113 .68515 Mdot
.48027 .3268 Mdot
.46682 .34598 Mdot
.49492 .30744 Mdot
.38864 .73208 Mdot
.37886 .70861 Mdot
.60072 .182 Mdot
.50467 .86415 Mdot
.49087 .85452 Mdot
.51737 .87233 Mdot
.61667 .16354 Mdot
.60915 .17218 Mdot
.56843 .21964 Mdot
.46058 .82978 Mdot
.44466 .81424 Mdot
.47614 .84315 Mdot
.59107 .19308 Mdot
.58036 .20563 Mdot
.35653 .52452 Mdot
.4099 .4708 Mdot
.40816 .48307 Mdot
.41259 .45808 Mdot
.35617 .55205 Mdot
.35633 .5379 Mdot
.35665 .4862 Mdot
.40742 .50698 Mdot
.40839 .519 Mdot
.40734 .49508 Mdot
.35667 .51163 Mdot
.35671 .49894 Mdot
.35868 .61999 Mdot
.44409 .38254 Mdot
.43501 .39954 Mdot
.45474 .36464 Mdot
.36535 .66229 Mdot
.3613 .64047 Mdot
.35615 .56716 Mdot
.42127 .43057 Mdot
.41633 .44473 Mdot
.42745 .41554 Mdot
.35716 .60098 Mdot
.35642 .58342 Mdot
.83979 .19794 Mdot
.38475 .97269 Mdot
.45508 .97553 Mdot
.27523 .93968 Mdot
.92766 .31029 Mdot
.88115 .24102 Mdot
.75316 .14245 Mdot
.52858 .9607 Mdot
.54754 .95183 Mdot
.49998 .96929 Mdot
.80558 .17105 Mdot
.77685 .1535 Mdot
.89444 .74201 Mdot
.35774 .03213 Mdot
.23503 .08243 Mdot
.43806 .02383 Mdot
.81474 .82246 Mdot
.85124 .79138 Mdot
.967 .41596 Mdot
.02578 .54808 Mdot
.12782 .81954 Mdot
.08681 .24357 Mdot
.94091 .66279 Mdot
.97333 .54705 Mdot
.64378 .13937 Mdot
.56122 .90128 Mdot
.55494 .89631 Mdot
.56624 .90623 Mdot
.65883 .13244 Mdot
.65088 .13546 Mdot
.62366 .15604 Mdot
.53893 .88552 Mdot
.52883 .87935 Mdot
.54755 .89112 Mdot
.63714 .14412 Mdot
.63053 .14966 Mdot
.701 .12885 Mdot
.56671 .93588 Mdot
.57094 .92887 Mdot
.55938 .94352 Mdot
.73286 .13527 Mdot
.7156 .13098 Mdot
.6674 .13009 Mdot
.57173 .91679 Mdot
.56975 .91136 Mdot
.57232 .92256 Mdot
.68812 .12808 Mdot
.67697 .12854 Mdot
.52168 .87495 Mdot
.61937 .1608 Mdot
.50951 .8672 Mdot
.61183 .16891 Mdot
.62623 .15369 Mdot
.54203 .88761 Mdot
.53244 .88168 Mdot
.58447 .20104 Mdot
.48143 .84743 Mdot
.46605 .83481 Mdot
.49602 .85814 Mdot
.60371 .17829 Mdot
.5947 .18899 Mdot
.56747 .90808 Mdot
.66162 .13142 Mdot
.65373 .13433 Mdot
.67062 .1294 Mdot
.57233 .91875 Mdot
.57064 .91325 Mdot
.55042 .89299 Mdot
.63937 .14228 Mdot
.63276 .14752 Mdot
.64642 .13796 Mdot
.56313 .90305 Mdot
.55751 .89806 Mdot
.45884 .35805 Mdot
.36718 .67035 Mdot
.36261 .64816 Mdot
.37358 .69346 Mdot
.48544 .3199 Mdot
.47153 .33917 Mdot
.42998 .40993 Mdot
.35762 .60762 Mdot
.35659 .5895 Mdot
.35953 .62719 Mdot
.44771 .37624 Mdot
.43812 .39356 Mdot
.54598 .24561 Mdot
.43456 .80307 Mdot
.41933 .78392 Mdot
.45027 .82005 Mdot
.57279 .21445 Mdot
.55992 .22933 Mdot
.50024 .30053 Mdot
.39264 .74035 Mdot
.38205 .717 Mdot
.40519 .76286 Mdot
.5311 .26307 Mdot
.51561 .28146 Mdot
.80337 .83049 Mdot
.50217 .03125 Mdot
.83733 .80423 Mdot
.45882 .02472 Mdot
.53031 .03993 Mdot
.75169 .85815 Mdot
.77523 .84733 Mdot
.13799 .16782 Mdot
.92464 .69524 Mdot
.96511 .59221 Mdot
.87829 .76238 Mdot
.39074 .02652 Mdot
.28408 .05624 Mdot
.68725 .87197 Mdot
.57223 .07782 Mdot
.57099 .07155 Mdot
.5718 .08363 Mdot
.66683 .86977 Mdot
.67642 .87132 Mdot
.73155 .86515 Mdot
.55995 .057 Mdot
.54841 .04873 Mdot
.56728 .06467 Mdot
.69999 .8713 Mdot
.71469 .86914 Mdot
.55577 .94642 Mdot
.73971 .13743 Mdot
.72163 .13235 Mdot
.76109 .14579 Mdot
.51992 .96384 Mdot
.54157 .95499 Mdot
.572 .92477 Mdot
.69247 .12818 Mdot
.68096 .12835 Mdot
.70582 .12933 Mdot
.56468 .93849 Mdot
.56961 .93133 Mdot
.2252 .91118 Mdot
.94378 .34377 Mdot
.89757 .26227 Mdot
.97429 .46166 Mdot
.02806 .43003 Mdot
.07817 .73969 Mdot
.48604 .972 Mdot
.81695 .17919 Mdot
.78662 .159 Mdot
.85381 .21112 Mdot
.3509 .96633 Mdot
.43361 .97619 Mdot
.64245 .85985 Mdot
.55374 .10466 Mdot
.64967 .8638 Mdot
.56027 .09967 Mdot
.54595 .10988 Mdot
.62931 .84921 Mdot
.6358 .85497 Mdot
.57153 .08427 Mdot
.66561 .86961 Mdot
.67507 .87126 Mdot
.65732 .86706 Mdot
.56538 .0947 Mdot
.56904 .08958 Mdot
.59902 .81604 Mdot
.48814 .14746 Mdot
.50225 .13754 Mdot
.47315 .15918 Mdot
.5782 .79189 Mdot
.5893 .80464 Mdot
.62253 .84258 Mdot
.52685 .1219 Mdot
.53702 .11555 Mdot
.51521 .12912 Mdot
.60755 .82614 Mdot
.61526 .83498 Mdot
.40039 .02547 Mdot
.83297 .80799 Mdot
.87289 .76867 Mdot
.7998 .83286 Mdot
.50634 .03224 Mdot
.46506 .02525 Mdot
.03417 .39177 Mdot
.96136 .60651 Mdot
.97553 .47747 Mdot
.91893 .7052 Mdot
.29911 .04995 Mdot
.15607 .14745 Mdot
.56766 .06547 Mdot
.69838 .87144 Mdot
.71264 .86961 Mdot
.68608 .87184 Mdot
.57247 .07859 Mdot
.57128 .07234 Mdot
.53276 .04097 Mdot
.74904 .85922 Mdot
.77215 .84892 Mdot
.72959 .86563 Mdot
.56117 .05799 Mdot
.55011 .04976 Mdot
.4095 .52689 Mdot
.35656 .47794 Mdot
.35637 .46463 Mdot
.35668 .49078 Mdot
.40729 .50272 Mdot
.40794 .5147 Mdot
.42026 .56673 Mdot
.35614 .43574 Mdot
.35631 .41975 Mdot
.3562 .45063 Mdot
.412 .5395 Mdot
.41557 .55275 Mdot
.41382 .45338 Mdot
.35614 .55733 Mdot
.35628 .5429 Mdot
.35617 .57278 Mdot
.4233 .42532 Mdot
.41797 .43976 Mdot
.40754 .49079 Mdot
.35663 .51617 Mdot
.35671 .50342 Mdot
.35647 .52925 Mdot
.41073 .46633 Mdot
.40867 .47873 Mdot
.52265 .72706 Mdot
.39809 .24925 Mdot
.41142 .22733 Mdot
.3866 .27229 Mdot
.49214 .68895 Mdot
.50724 .70824 Mdot
.56596 .77764 Mdot
.44176 .18891 Mdot
.45759 .17295 Mdot
.42617 .20709 Mdot
.53799 .74504 Mdot
.55254 .76196 Mdot
.44232 .61423 Mdot
.35834 .38366 Mdot
.36075 .36342 Mdot
.35696 .40243 Mdot
.42618 .58159 Mdot
.43349 .59742 Mdot
.47772 .6696 Mdot
.36991 .31917 Mdot
.3772 .29581 Mdot
.36451 .34184 Mdot
.45262 .63197 Mdot
.46443 .6505 Mdot
.75846 .14466 Mdot
.52281 .96281 Mdot
.73743 .1367 Mdot
.54378 .95391 Mdot
.49107 .97113 Mdot
.81316 .17641 Mdot
.78319 .15698 Mdot
.57023 .93047 Mdot
.70431 .12922 Mdot
.69103 .12815 Mdot
.71945 .13177 Mdot
.55703 .94545 Mdot
.56528 .93764 Mdot
.97247 .44637 Mdot
.02458 .46905 Mdot
.09378 .76884 Mdot
.12015 .19069 Mdot
.93005 .68521 Mdot
.96842 .57734 Mdot
.84906 .20651 Mdot
.36274 .96892 Mdot
.44119 .97612 Mdot
.24255 .92219 Mdot
.9385 .33193 Mdot
.89192 .25462 Mdot
.53135 .88089 Mdot
.62529 .15442 Mdot
.61838 .16166 Mdot
.63211 .14826 Mdot
.54944 .89238 Mdot
.54114 .88689 Mdot
.49427 .85696 Mdot
.60282 .17955 Mdot
.59344 .1903 Mdot
.61103 .17002 Mdot
.5202 .8741 Mdot
.50779 .86622 Mdot
.57037 .91262 Mdot
.66959 .12965 Mdot
.66081 .13181 Mdot
.67947 .12832 Mdot
.57217 .92402 Mdot
.57201 .91813 Mdot
.55653 .89751 Mdot
.64543 .13837 Mdot
.63873 .14293 Mdot
.65267 .13465 Mdot
.56722 .90743 Mdot
.56252 .90246 Mdot
.53927 .11415 Mdot
.62417 .84425 Mdot
.5479 .1086 Mdot
.63088 .85065 Mdot
.617 .8369 Mdot
.5181 .1273 Mdot
.52943 .12032 Mdot
.65144 .86464 Mdot
.56162 .09849 Mdot
.56641 .0935 Mdot
.55539 .10345 Mdot
.63738 .8562 Mdot
.64409 .86088 Mdot
.46136 .16947 Mdot
.569 .78116 Mdot
.58094 .79507 Mdot
.55585 .76581 Mdot
.42982 .20257 Mdot
.44554 .1849 Mdot
.50543 .13541 Mdot
.60116 .81855 Mdot
.60946 .82835 Mdot
.59173 .80748 Mdot
.4768 .15621 Mdot
.49158 .14494 Mdot
.77835 .84567 Mdot
.52746 .03886 Mdot
.49806 .0303 Mdot
.5466 .04769 Mdot
.73398 .86435 Mdot
.75417 .85719 Mdot
.88345 .75617 Mdot
.38019 .02799 Mdot
.26839 .06365 Mdot
.4523 .0243 Mdot
.8071 .82793 Mdot
.84194 .8001 Mdot
.67753 .87151 Mdot
.57188 .08295 Mdot
.57241 .07716 Mdot
.56975 .08834 Mdot
.65924 .86772 Mdot
.66775 .87008 Mdot
.7164 .86889 Mdot
.56641 .06375 Mdot
.55902 .0561 Mdot
.57065 .07076 Mdot
.68888 .87184 Mdot
.70159 .87115 Mdot
.38796 .26948 Mdot
.49383 .69128 Mdot
.39967 .24654 Mdot
.50913 .7105 Mdot
.47936 .6719 Mdot
.37063 .31644 Mdot
.37826 .29299 Mdot
.5541 .76393 Mdot
.42787 .20483 Mdot
.44351 .1869 Mdot
.4131 .22482 Mdot
.52458 .72922 Mdot
.53968 .74714 Mdot
.35712 .40022 Mdot
.42701 .58343 Mdot
.43448 .59937 Mdot
.42088 .56844 Mdot
.35612 .43393 Mdot
.35638 .41774 Mdot
.36499 .33924 Mdot
.45395 .63414 Mdot
.46601 .65275 Mdot
.44342 .61629 Mdot
.35859 .3813 Mdot
.36108 .36095 Mdot
.63668 .85555 Mdot
.54714 .10928 Mdot
.55461 .10406 Mdot
.53822 .11486 Mdot
.62326 .84347 Mdot
.62991 .85002 Mdot
.65809 .86749 Mdot
.56582 .09408 Mdot
.56961 .08902 Mdot
.56075 .09903 Mdot
.64345 .86028 Mdot
.65044 .86429 Mdot
.59041 .80612 Mdot
.47509 .15769 Mdot
.49004 .1462 Mdot
.45943 .1712 Mdot
.5676 .77937 Mdot
.5797 .79344 Mdot
.6163 .83587 Mdot
.51651 .12818 Mdot
.52798 .12107 Mdot
.50389 .13647 Mdot
.59996 .81737 Mdot
.60862 .82721 Mdot
.37279 .69071 Mdot
.48366 .32219 Mdot
.46991 .34145 Mdot
.49851 .30284 Mdot
.39129 .73761 Mdot
.38101 .71423 Mdot
.35919 .62474 Mdot
.44649 .37835 Mdot
.43703 .39558 Mdot
.4575 .36025 Mdot
.36655 .66765 Mdot
.36212 .64556 Mdot
.44845 .81814 Mdot
.5714 .21618 Mdot
.55844 .23125 Mdot
.58303 .20251 Mdot
.47976 .84603 Mdot
.46434 .83316 Mdot
.40352 .76023 Mdot
.52918 .26519 Mdot
.51384 .28372 Mdot
.54418 .24761 Mdot
.43262 .80089 Mdot
.41745 .78148 Mdot
.35668 .48925 Mdot
.40732 .50414 Mdot
.40807 .51611 Mdot
.40746 .49224 Mdot
.35664 .51468 Mdot
.35671 .50195 Mdot
.35617 .44895 Mdot
.41239 .54107 Mdot
.41606 .55436 Mdot
.40974 .52837 Mdot
.35655 .47633 Mdot
.35635 .46299 Mdot
.35618 .57092 Mdot
.42263 .42707 Mdot
.4174 .44144 Mdot
.42913 .41181 Mdot
.35745 .60538 Mdot
.35655 .58748 Mdot
.35648 .52766 Mdot
.41045 .46781 Mdot
.40849 .48017 Mdot
.41338 .45497 Mdot
.35614 .55555 Mdot
.35628 .54119 Mdot
.58611 .19887 Mdot
.48391 .84934 Mdot
.57481 .21211 Mdot
.46879 .83704 Mdot
.49821 .85976 Mdot
.60525 .1767 Mdot
.59632 .18713 Mdot
.42171 .78723 Mdot
.54841 .24283 Mdot
.53351 .26006 Mdot
.56225 .2268 Mdot
.45303 .82266 Mdot
.43714 .80604 Mdot
.634 .14666 Mdot
.55154 .89388 Mdot
.54356 .88852 Mdot
.55842 .89892 Mdot
.64741 .13721 Mdot
.64054 .14152 Mdot
.61314 .16749 Mdot
.52357 .87612 Mdot
.51149 .86859 Mdot
.53431 .88267 Mdot
.62734 .15261 Mdot
.62036 .15947 Mdot
.35672 .59242 Mdot
.43122 .40729 Mdot
.42434 .42282 Mdot
.43957 .39076 Mdot
.3599 .63054 Mdot
.35784 .61074 Mdot
.35624 .54521 Mdot
.41441 .45119 Mdot
.41118 .46421 Mdot
.41878 .43742 Mdot
.35624 .57552 Mdot
.35613 .55983 Mdot
.3837 .72092 Mdot
.50285 .29735 Mdot
.48785 .31667 Mdot
.5182 .27835 Mdot
.40734 .76646 Mdot
.39454 .74416 Mdot
.36324 .65175 Mdot
.46088 .35495 Mdot
.44949 .37327 Mdot
.47369 .33599 Mdot
.37492 .69739 Mdot
.36816 .67416 Mdot
.16335 .13996 Mdot
.91679 .70876 Mdot
.03727 .37697 Mdot
.95977 .61202 Mdot
.87093 .77086 Mdot
.4042 .02511 Mdot
.30487 .04772 Mdot
.90537 .27341 Mdot
.20041 .89311 Mdot
.33328 .96171 Mdot
.05888 .69527 Mdot
.97585 .48362 Mdot
.95072 .36109 Mdot
.55085 .05018 Mdot
.72861 .86595 Mdot
.74815 .85953 Mdot
.71195 .86971 Mdot
.5679 .06581 Mdot
.56146 .05833 Mdot
.46734 .02548 Mdot
.79846 .83371 Mdot
.83116 .80953 Mdot
.77091 .84957 Mdot
.5337 .04138 Mdot
.50785 .03264 Mdot
.68261 .12816 Mdot
.57192 .92578 Mdot
.57229 .91975 Mdot
.56904 .93247 Mdot
.70833 .12972 Mdot
.6947 .12836 Mdot
.65502 .13381 Mdot
.56825 .90889 Mdot
.56405 .90385 Mdot
.57091 .91418 Mdot
.67221 .12914 Mdot
.6632 .13111 Mdot
.79133 .16181 Mdot
.47888 .97312 Mdot
.51519 .9653 Mdot
.42249 .976 Mdot
.86071 .21813 Mdot
.8226 .18353 Mdot
.72426 .1329 Mdot
.55393 .94778 Mdot
.56334 .9398 Mdot
.53855 .95646 Mdot
.76499 .14757 Mdot
.74316 .13867 Mdot
.57138 .08522 Mdot
.66413 .86926 Mdot
.57234 .07954 Mdot
.67361 .87094 Mdot
.65584 .86665 Mdot
.56445 .09549 Mdot
.56861 .09047 Mdot
.71031 .86991 Mdot
.5686 .06669 Mdot
.56255 .05929 Mdot
.5715 .07338 Mdot
.68407 .87186 Mdot
.69608 .87168 Mdot
.53537 .11655 Mdot
.62138 .84138 Mdot
.62806 .84824 Mdot
.61419 .83351 Mdot
.51312 .13042 Mdot
.52483 .12299 Mdot
.55918 .10047 Mdot
.64151 .859 Mdot
.64846 .86319 Mdot
.63465 .85411 Mdot
.54468 .11081 Mdot
.55267 .10554 Mdot
.97619 .49969 Mdot
.04673 .34069 Mdot
.04718 .66081 Mdot
.18154 .12272 Mdot
.91107 .71796 Mdot
.95552 .62542 Mdot
.86555 .22322 Mdot
.31967 .9575 Mdot
.41369 .97557 Mdot
.18235 .87801 Mdot
.95531 .37395 Mdot
.91083 .28167 Mdot
.76818 .85085 Mdot
.53633 .04248 Mdot
.51138 .03362 Mdot
.55218 .05114 Mdot
.72653 .86648 Mdot
.74548 .86059 Mdot
.86579 .77653 Mdot
.41334 .02445 Mdot
.31912 .04269 Mdot
.47335 .02614 Mdot
.79473 .83614 Mdot
.82711 .81283 Mdot
.35625 .4225 Mdot
.41938 .56434 Mdot
.4251 .57905 Mdot
.4149 .5505 Mdot
.35622 .45302 Mdot
.35611 .43833 Mdot
.36027 .36687 Mdot
.44073 .61137 Mdot
.45077 .62896 Mdot
.43221 .59473 Mdot
.35683 .40537 Mdot
.35808 .38684 Mdot
.3567 .50557 Mdot
.40766 .48881 Mdot
.40727 .50075 Mdot
.40895 .47668 Mdot
.35642 .53144 Mdot
.3566 .51832 Mdot
.35641 .46685 Mdot
.40917 .52482 Mdot
.41153 .53739 Mdot
.40777 .51268 Mdot
.3567 .4929 Mdot
.35659 .48006 Mdot
.45486 .17545 Mdot
.5638 .77514 Mdot
.5764 .78957 Mdot
.55008 .75927 Mdot
.42361 .21032 Mdot
.439 .19179 Mdot
.5001 .13909 Mdot
.59736 .81429 Mdot
.60615 .82457 Mdot
.58754 .80263 Mdot
.47063 .16133 Mdot
.48584 .14929 Mdot
.37584 .29973 Mdot
.47545 .6664 Mdot
.48962 .68575 Mdot
.46241 .64738 Mdot
.36372 .34555 Mdot
.36882 .32304 Mdot
.40916 .23085 Mdot
.5202 .72395 Mdot
.53547 .74212 Mdot
.50467 .70506 Mdot
.38495 .27616 Mdot
.39613 .25299 Mdot
.56954 .93163 Mdot
.70664 .1295 Mdot
.57186 .92508 Mdot
.69298 .12816 Mdot
.72239 .13251 Mdot
.55511 .94683 Mdot
.5644 .93884 Mdot
.662 .1313 Mdot
.57085 .91348 Mdot
.5677 .90831 Mdot
.57231 .91904 Mdot
.68141 .12829 Mdot
.67122 .12941 Mdot
.43043 .97616 Mdot
.85564 .21293 Mdot
.81869 .18054 Mdot
.89968 .26519 Mdot
.21835 .90648 Mdot
.3459 .9651 Mdot
.54085 .95538 Mdot
.76235 .14641 Mdot
.74056 .13769 Mdot
.78783 .15968 Mdot
.48418 .97232 Mdot
.51851 .96427 Mdot
.59508 .18842 Mdot
.49673 .85859 Mdot
.48221 .84797 Mdot
.51009 .8676 Mdot
.61231 .16856 Mdot
.60409 .17781 Mdot
.56055 .2286 Mdot
.45099 .82079 Mdot
.4352 .80391 Mdot
.46682 .83544 Mdot
.58496 .20043 Mdot
.57347 .21382 Mdot
.63982 .14212 Mdot
.55775 .89831 Mdot
.55085 .89322 Mdot
.56327 .90331 Mdot
.65399 .13413 Mdot
.64675 .13777 Mdot
.6197 .16045 Mdot
.53292 .88197 Mdot
.52209 .87531 Mdot
.54253 .88785 Mdot
.63303 .14723 Mdot
.62646 .15335 Mdot
.65019 .86407 Mdot
.56056 .09928 Mdot
.65773 .86734 Mdot
.56554 .0943 Mdot
.55436 .10432 Mdot
.63623 .85537 Mdot
.64314 .86007 Mdot
.57232 .07812 Mdot
.676 .87124 Mdot
.68679 .87192 Mdot
.66626 .86975 Mdot
.56938 .08924 Mdot
.57179 .08391 Mdot
.60811 .82683 Mdot
.50333 .1369 Mdot
.51605 .12855 Mdot
.48932 .1467 Mdot
.59001 .80552 Mdot
.59954 .81686 Mdot
.62963 .84972 Mdot
.53769 .11512 Mdot
.54669 .10952 Mdot
.52745 .12138 Mdot
.61597 .83548 Mdot
.62302 .8431 Mdot
.2902 .05359 Mdot
.8762 .76485 Mdot
.9224 .69922 Mdot
.83568 .80565 Mdot
.46146 .02492 Mdot
.39465 .02606 Mdot
.07244 .72767 Mdot
.97488 .4681 Mdot
.94572 .34839 Mdot
.96373 .59773 Mdot
.14506 .15956 Mdot
.03019 .41467 Mdot
.56052 .05741 Mdot
.71398 .86926 Mdot
.73078 .86533 Mdot
.69924 .87143 Mdot
.57101 .07184 Mdot
.56748 .065 Mdot
.50371 .03162 Mdot
.77412 .84788 Mdot
.80185 .83154 Mdot
.75052 .85867 Mdot
.54898 .04912 Mdot
.53138 .04036 Mdot
.55189 .76119 Mdot
.4254 .20801 Mdot
.56542 .77691 Mdot
.44099 .18972 Mdot
.41071 .22834 Mdot
.52194 .72619 Mdot
.5372 .74424 Mdot
.48747 .14797 Mdot
.5887 .80412 Mdot
.59861 .81552 Mdot
.57766 .79125 Mdot
.4569 .17365 Mdot
.47253 .15978 Mdot
.46386 .64962 Mdot
.36428 .3429 Mdot
.36962 .32025 Mdot
.36059 .36443 Mdot
.44185 .61342 Mdot
.45213 .63112 Mdot
.50656 .70733 Mdot
.38614 .27338 Mdot
.39749 .25031 Mdot
.37687 .29689 Mdot
.47702 .6687 Mdot
.49145 .68804 Mdot
.569 .08985 Mdot
.65696 .86691 Mdot
.66529 .86946 Mdot
.64918 .8637 Mdot
.55991 .09989 Mdot
.56522 .09495 Mdot
.57149 .07267 Mdot
.68543 .87189 Mdot
.69785 .87144 Mdot
.67459 .87121 Mdot
.57138 .08451 Mdot
.57239 .07884 Mdot
.52632 .12221 Mdot
.61493 .83458 Mdot
.62207 .84231 Mdot
.60725 .82566 Mdot
.50155 .13796 Mdot
.51455 .12948 Mdot
.55333 .10488 Mdot
.63558 .85468 Mdot
.64216 .85962 Mdot
.629 .84892 Mdot
.5367 .11586 Mdot
.54559 .11014 Mdot
.43852 .39277 Mdot
.35964 .62815 Mdot
.3577 .60852 Mdot
.36275 .64915 Mdot
.45943 .35717 Mdot
.44818 .37541 Mdot
.41818 .43911 Mdot
.35618 .57354 Mdot
.35612 .558 Mdot
.35664 .59034 Mdot
.43036 .40918 Mdot
.42358 .42462 Mdot
.51642 .2806 Mdot
.40584 .7639 Mdot
.39323 .74145 Mdot
.42 .78486 Mdot
.54661 .2448 Mdot
.53181 .26222 Mdot
.47217 .33828 Mdot
.37394 .69456 Mdot
.36741 .6714 Mdot
.38254 .71812 Mdot
.50094 .29962 Mdot
.48608 .31898 Mdot
.41536 .5521 Mdot
.3562 .45133 Mdot
.35614 .43646 Mdot
.35638 .46528 Mdot
.40941 .52631 Mdot
.41187 .5389 Mdot
.43311 .59665 Mdot
.35692 .40326 Mdot
.35824 .38459 Mdot
.35631 .4205 Mdot
.42002 .56606 Mdot
.42589 .58087 Mdot
.40874 .47817 Mdot
.35646 .5299 Mdot
.35662 .51679 Mdot
.35626 .54355 Mdot
.41399 .45275 Mdot
.41086 .46572 Mdot
.40789 .51413 Mdot
.35669 .49135 Mdot
.35656 .47854 Mdot
.35671 .50402 Mdot
.40757 .49024 Mdot
.40728 .50215 Mdot
.43856 .60732 Mdot
.35771 .39138 Mdot
.44823 .62469 Mdot
.35965 .37176 Mdot
.35665 .40957 Mdot
.42362 .57546 Mdot
.4304 .59091 Mdot
.37399 .30532 Mdot
.47223 .66183 Mdot
.48615 .68113 Mdot
.45951 .64293 Mdot
.36276 .35075 Mdot
.36744 .32849 Mdot
.40875 .5219 Mdot
.35662 .48314 Mdot
.35646 .47004 Mdot
.35671 .49592 Mdot
.40728 .49792 Mdot
.40757 .50983 Mdot
.4182 .56097 Mdot
.35612 .44193 Mdot
.35619 .42638 Mdot
.35626 .45639 Mdot
.41088 .53435 Mdot
.414 .54732 Mdot
.49681 .14137 Mdot
.59511 .81165 Mdot
.60414 .82224 Mdot
.585 .79965 Mdot
.46692 .16449 Mdot
.4823 .15198 Mdot
.53298 .11801 Mdot
.61972 .83961 Mdot
.62647 .84671 Mdot
.61238 .83148 Mdot
.51011 .13236 Mdot
.52213 .12466 Mdot
.4059 .236 Mdot
.51651 .7195 Mdot
.53187 .73788 Mdot
.50102 .70048 Mdot
.3826 .28176 Mdot
.3933 .25844 Mdot
.45107 .17914 Mdot
.56063 .77148 Mdot
.57354 .78625 Mdot
.54666 .7553 Mdot
.42004 .21505 Mdot
.43526 .19601 Mdot
.44089 .8102 Mdot
.56537 .22318 Mdot
.4253 .7919 Mdot
.55182 .2389 Mdot
.57762 .20883 Mdot
.47248 .84016 Mdot
.45681 .82628 Mdot
.5065 .29278 Mdot
.39743 .74958 Mdot
.38611 .72652 Mdot
.41063 .77156 Mdot
.5371 .25585 Mdot
.52187 .27392 Mdot
.52624 .87777 Mdot
.62202 .15773 Mdot
.61492 .16548 Mdot
.62894 .15111 Mdot
.54559 .88983 Mdot
.53666 .88412 Mdot
.48742 .85198 Mdot
.59855 .18453 Mdot
.58862 .19594 Mdot
.60723 .1744 Mdot
.51447 .87049 Mdot
.50147 .862 Mdot
.41534 .44798 Mdot
.35613 .56346 Mdot
.3562 .5486 Mdot
.35632 .57943 Mdot
.42586 .41921 Mdot
.41999 .43402 Mdot
.40788 .48595 Mdot
.35657 .52141 Mdot
.35669 .5086 Mdot
.35638 .53465 Mdot
.41185 .46117 Mdot
.4094 .47375 Mdot
.46381 .35048 Mdot
.36959 .67964 Mdot
.36425 .65699 Mdot
.37683 .703 Mdot
.49136 .31206 Mdot
.47695 .3314 Mdot
.43307 .40344 Mdot
.35823 .61532 Mdot
.35692 .59666 Mdot
.36057 .63547 Mdot
.45207 .36898 Mdot
.44179 .38668 Mdot
.56551 .90568 Mdot
.65771 .13269 Mdot
.56059 .90068 Mdot
.65017 .13597 Mdot
.66619 .13025 Mdot
.57177 .91607 Mdot
.56933 .91075 Mdot
.62962 .15033 Mdot
.54662 .89047 Mdot
.53762 .88486 Mdot
.55433 .89566 Mdot
.64308 .13995 Mdot
.63618 .14466 Mdot
.56061 .94254 Mdot
.73072 .13467 Mdot
.71388 .13071 Mdot
.75039 .14128 Mdot
.53148 .95961 Mdot
.54904 .95085 Mdot
.57238 .92184 Mdot
.68676 .12811 Mdot
.67595 .12877 Mdot
.69917 .12857 Mdot
.56747 .93498 Mdot
.57101 .92814 Mdot
.53898 .25379 Mdot
.42718 .7942 Mdot
.41243 .7741 Mdot
.4427 .81223 Mdot
.56681 .22136 Mdot
.55338 .23692 Mdot
.49308 .30974 Mdot
.38736 .7293 Mdot
.37776 .70577 Mdot
.399 .75229 Mdot
.52379 .27174 Mdot
.50824 .29048 Mdot
.60805 .17322 Mdot
.51603 .87141 Mdot
.50326 .86306 Mdot
.52741 .87859 Mdot
.62301 .15695 Mdot
.61591 .16455 Mdot
.57912 .20727 Mdot
.4742 .84167 Mdot
.45855 .82805 Mdot
.48922 .85325 Mdot
.59952 .1832 Mdot
.58998 .19455 Mdot
.56955 .06835 Mdot
.69303 .87185 Mdot
.70675 .87046 Mdot
.68144 .87174 Mdot
.57226 .08093 Mdot
.57186 .07489 Mdot
.54079 .04459 Mdot
.74065 .86229 Mdot
.76248 .85353 Mdot
.72244 .86749 Mdot
.56435 .06112 Mdot
.55505 .05313 Mdot
.55773 .10166 Mdot
.63989 .8579 Mdot
.64677 .86226 Mdot
.63307 .8528 Mdot
.5426 .11213 Mdot
.5509 .10677 Mdot
.57087 .0865 Mdot
.66206 .86871 Mdot
.67129 .8706 Mdot
.654 .8659 Mdot
.56329 .09667 Mdot
.56775 .09168 Mdot
.29084 .94669 Mdot
.92212 .3003 Mdot
.87594 .23485 Mdot
.96358 .4017 Mdot
.03044 .58695 Mdot
.14581 .8413 Mdot
.50389 .96834 Mdot
.8017 .16837 Mdot
.774 .15207 Mdot
.83552 .19422 Mdot
.39504 .97399 Mdot
.46172 .97507 Mdot
.43011 .02385 Mdot
.81888 .81932 Mdot
.85584 .78687 Mdot
.78794 .84027 Mdot
.51832 .03569 Mdot
.484 .02765 Mdot
.07184 .27364 Mdot
.94593 .65112 Mdot
.97493 .53123 Mdot
.89989 .73452 Mdot
.34537 .03504 Mdot
.21763 .09404 Mdot
.35625 .57742 Mdot
.42506 .42103 Mdot
.35611 .5616 Mdot
.41936 .43574 Mdot
.43217 .40536 Mdot
.35807 .61307 Mdot
.35682 .59454 Mdot
.40916 .47525 Mdot
.35642 .53309 Mdot
.35659 .51987 Mdot
.35622 .54692 Mdot
.41489 .44957 Mdot
.41151 .46268 Mdot
.37579 .70016 Mdot
.48956 .31436 Mdot
.47539 .33371 Mdot
.50458 .29504 Mdot
.39606 .7469 Mdot
.38489 .72372 Mdot
.36027 .63303 Mdot
.45071 .37114 Mdot
.44069 .38872 Mdot
.46234 .35272 Mdot
.36879 .67685 Mdot
.36371 .65435 Mdot
.43125 .5928 Mdot
.35672 .4075 Mdot
.35784 .38917 Mdot
.35624 .42441 Mdot
.4188 .56266 Mdot
.42437 .57726 Mdot
.46093 .64515 Mdot
.36327 .34814 Mdot
.36819 .32572 Mdot
.35992 .36936 Mdot
.43963 .60934 Mdot
.44954 .62682 Mdot
.40766 .51126 Mdot
.3567 .49437 Mdot
.3566 .48163 Mdot
.3567 .50704 Mdot
.40777 .48738 Mdot
.40727 .49931 Mdot
.41443 .54889 Mdot
.35624 .45472 Mdot
.35614 .44009 Mdot
.35642 .4685 Mdot
.40896 .52338 Mdot
.41119 .53585 Mdot
.67355 .12906 Mdot
.5724 .92043 Mdot
.66406 .13075 Mdot
.57135 .91477 Mdot
.5715 .9266 Mdot
.69602 .12833 Mdot
.68404 .12817 Mdot
.5592 .8995 Mdot
.64845 .13685 Mdot
.64145 .14101 Mdot
.65583 .13339 Mdot
.56856 .90952 Mdot
.56443 .90449 Mdot
.76806 .1491 Mdot
.51156 .96634 Mdot
.53641 .95749 Mdot
.47356 .97384 Mdot
.82693 .18703 Mdot
.7946 .16379 Mdot
.7102 .13006 Mdot
.56262 .94067 Mdot
.56859 .93329 Mdot
.55223 .94883 Mdot
.74537 .13938 Mdot
.72648 .13352 Mdot
.45477 .82448 Mdot
.57633 .2105 Mdot
.56371 .22494 Mdot
.58751 .19745 Mdot
.48574 .85066 Mdot
.47053 .83862 Mdot
.40909 .76904 Mdot
.53541 .25798 Mdot
.52011 .27615 Mdot
.55002 .24082 Mdot
.43894 .80813 Mdot
.42356 .78959 Mdot
.53531 .88343 Mdot
.62805 .15181 Mdot
.62137 .15867 Mdot
.6346 .14592 Mdot
.55263 .89445 Mdot
.54461 .88917 Mdot
.50003 .86087 Mdot
.60609 .17548 Mdot
.59733 .18578 Mdot
.61413 .16652 Mdot
.52479 .87698 Mdot
.51309 .86954 Mdot
.9596 .38744 Mdot
.03764 .62465 Mdot
.91657 .29089 Mdot
.16412 .86081 Mdot
.05836 .30612 Mdot
.95091 .63841 Mdot
.97587 .5157 Mdot
.46759 .9745 Mdot
.83098 .19032 Mdot
.79831 .1662 Mdot
.87074 .22893 Mdot
.30551 .95252 Mdot
.40465 .97493 Mdot
.79149 .83809 Mdot
.51508 .03467 Mdot
.47863 .02686 Mdot
.53843 .0435 Mdot
.74325 .86131 Mdot
.76509 .85239 Mdot
.90561 .72624 Mdot
.33271 .03846 Mdot
.19966 .10749 Mdot
.42212 .02402 Mdot
.82275 .81636 Mdot
.8609 .78168 Mdot
.56904 .91011 Mdot
.66527 .13057 Mdot
.65691 .1331 Mdot
.67455 .12881 Mdot
.57236 .92114 Mdot
.57136 .91547 Mdot
.55328 .8951 Mdot
.64215 .14042 Mdot
.63557 .14537 Mdot
.64913 .13631 Mdot
.56518 .90504 Mdot
.55985 .9001 Mdot
.55091 .94979 Mdot
.74807 .14046 Mdot
.72847 .134 Mdot
.77079 .15038 Mdot
.50799 .96733 Mdot
.53381 .95858 Mdot
.5715 .9273 Mdot
.6978 .12857 Mdot
.68532 .12809 Mdot
.7119 .1303 Mdot
.56148 .94164 Mdot
.56796 .93415 Mdot
.58618 .80119 Mdot
.46884 .16289 Mdot
.48396 .15061 Mdot
.45311 .17727 Mdot
.5623 .77329 Mdot
.57485 .78797 Mdot
.61315 .83257 Mdot
.51158 .13138 Mdot
.52365 .12385 Mdot
.49829 .1402 Mdot
.59638 .81292 Mdot
.60527 .82336 Mdot
.50292 .70276 Mdot
.38372 .27897 Mdot
.3946 .25574 Mdot
.37495 .3025 Mdot
.47376 .66412 Mdot
.48794 .68343 Mdot
.5485 .75725 Mdot
.42181 .21268 Mdot
.43724 .19388 Mdot
.40742 .23344 Mdot
.51826 .72176 Mdot
.5336 .74003 Mdot
.68271 .87183 Mdot
.57232 .08024 Mdot
.57192 .0742 Mdot
.57092 .0858 Mdot
.66322 .86892 Mdot
.67224 .87089 Mdot
.72438 .86707 Mdot
.56333 .06018 Mdot
.55388 .05219 Mdot
.56898 .06749 Mdot
.69476 .87165 Mdot
.70837 .87029 Mdot
.63401 .85338 Mdot
.54356 .11145 Mdot
.55159 .1061 Mdot
.53435 .1173 Mdot
.62041 .84057 Mdot
.6274 .84742 Mdot
.65508 .8662 Mdot
.56409 .09613 Mdot
.56821 .09108 Mdot
.55848 .10107 Mdot
.64055 .85853 Mdot
.64746 .86281 Mdot
.55625 .89727 Mdot
.64519 .13861 Mdot
.5492 .8921 Mdot
.63845 .14318 Mdot
.65231 .1348 Mdot
.56696 .90721 Mdot
.56218 .90225 Mdot
.61813 .16206 Mdot
.53077 .88061 Mdot
.51959 .87377 Mdot
.54068 .88663 Mdot
.63167 .14847 Mdot
.62497 .15474 Mdot
.57034 .93016 Mdot
.70366 .12915 Mdot
.69034 .12807 Mdot
.71877 .13167 Mdot
.55743 .94508 Mdot
.56574 .93725 Mdot
.57035 .91234 Mdot
.66922 .12977 Mdot
.66028 .13186 Mdot
.67905 .1284 Mdot
.57212 .92375 Mdot
.57201 .91785 Mdot
.51314 .28461 Mdot
.40297 .75921 Mdot
.3908 .73654 Mdot
.41678 .78053 Mdot
.54347 .2484 Mdot
.52852 .26606 Mdot
.46933 .34235 Mdot
.37242 .68959 Mdot
.36628 .66657 Mdot
.38057 .71311 Mdot
.49773 .30373 Mdot
.48299 .3231 Mdot
.59297 .19084 Mdot
.49372 .85648 Mdot
.47905 .84547 Mdot
.50722 .86582 Mdot
.6107 .17046 Mdot
.6023 .17998 Mdot
.55772 .23196 Mdot
.44763 .81738 Mdot
.43185 .80002 Mdot
.46354 .8325 Mdot
.58257 .20314 Mdot
.57087 .21687 Mdot
.7173 .86864 Mdot
.56623 .06342 Mdot
.7348 .86412 Mdot
.55859 .05572 Mdot
.57043 .07042 Mdot
.68938 .87187 Mdot
.70216 .87113 Mdot
.49617 .02992 Mdot
.77967 .84492 Mdot
.80851 .82697 Mdot
.75516 .8568 Mdot
.54579 .04727 Mdot
.52645 .03845 Mdot
.65182 .86479 Mdot
.56186 .09826 Mdot
.56646 .09323 Mdot
.55579 .10324 Mdot
.63763 .85647 Mdot
.64453 .86102 Mdot
.67818 .87147 Mdot
.57204 .08271 Mdot
.57235 .07686 Mdot
.56992 .0881 Mdot
.65953 .8679 Mdot
.66815 .87019 Mdot
.44398 .97604 Mdot
.84713 .20467 Mdot
.81167 .17534 Mdot
.88982 .25189 Mdot
.24912 .92603 Mdot
.36725 .9698 Mdot
.54453 .9535 Mdot
.75745 .14425 Mdot
.73639 .13632 Mdot
.78194 .15628 Mdot
.49285 .97078 Mdot
.52403 .9624 Mdot
.26201 .06691 Mdot
.88571 .75335 Mdot
.93232 .68079 Mdot
.84374 .79846 Mdot
.44967 .02416 Mdot
.37604 .02864 Mdot
.10018 .77955 Mdot
.97155 .44016 Mdot
.93642 .32752 Mdot
.96954 .57155 Mdot
.11335 .20027 Mdot
.02395 .48477 Mdot
.61656 .83629 Mdot
.51709 .12782 Mdot
.62352 .84382 Mdot
.52857 .12078 Mdot
.50441 .13603 Mdot
.60048 .81782 Mdot
.60902 .82763 Mdot
.55481 .10379 Mdot
.63704 .85577 Mdot
.64369 .86052 Mdot
.63032 .85026 Mdot
.53874 .1146 Mdot
.54748 .10902 Mdot
.55484 .76466 Mdot
.4286 .20394 Mdot
.44431 .18611 Mdot
.41371 .22385 Mdot
.52525 .7301 Mdot
.54035 .74797 Mdot
.59085 .80669 Mdot
.47585 .15711 Mdot
.49066 .1457 Mdot
.46024 .17051 Mdot
.5682 .78006 Mdot
.58017 .7941 Mdot
.54782 .04835 Mdot
.73242 .8649 Mdot
.75275 .85771 Mdot
.71534 .86906 Mdot
.56692 .0643 Mdot
.55952 .05663 Mdot
.45617 .02454 Mdot
.80496 .82938 Mdot
.83902 .80275 Mdot
.77635 .84677 Mdot
.52907 .03949 Mdot
.50065 .03088 Mdot
.56975 .08877 Mdot
.6586 .86755 Mdot
.66726 .86986 Mdot
.65073 .86448 Mdot
.56106 .09881 Mdot
.56612 .09387 Mdot
.57097 .07127 Mdot
.68782 .87197 Mdot
.70074 .87117 Mdot
.67681 .87143 Mdot
.57172 .08333 Mdot
.57227 .07755 Mdot
.40841 .48076 Mdot
.3565 .52707 Mdot
.35665 .51407 Mdot
.35629 .54055 Mdot
.41323 .45557 Mdot
.41034 .4684 Mdot
.40813 .51668 Mdot
.35667 .48863 Mdot
.35653 .47573 Mdot
.35671 .50134 Mdot
.40743 .49279 Mdot
.40734 .50469 Mdot
.43664 .39636 Mdot
.35911 .62382 Mdot
.35742 .60453 Mdot
.36194 .64454 Mdot
.45694 .36112 Mdot
.44598 .37919 Mdot
.41718 .44209 Mdot
.35616 .57015 Mdot
.35614 .55483 Mdot
.35651 .58667 Mdot
.4288 .41255 Mdot
.42234 .42778 Mdot
.46658 .65364 Mdot
.36524 .33817 Mdot
.37098 .31533 Mdot
.36123 .35996 Mdot
.44388 .61711 Mdot
.45451 .635 Mdot
.50987 .71139 Mdot
.38846 .26838 Mdot
.4002 .24551 Mdot
.37869 .29187 Mdot
.47999 .67281 Mdot
.49458 .69218 Mdot
.41625 .555 Mdot
.35616 .44826 Mdot
.35614 .43318 Mdot
.35633 .46238 Mdot
.40985 .52896 Mdot
.41253 .54167 Mdot
.43485 .60014 Mdot
.35716 .39937 Mdot
.35865 .38041 Mdot
.35641 .41693 Mdot
.42115 .56914 Mdot
.42732 .58416 Mdot
.88442 .24504 Mdot
.2657 .935 Mdot
.84272 .20062 Mdot
.37844 .97175 Mdot
.11724 .80528 Mdot
.96891 .42515 Mdot
.93103 .31668 Mdot
.52705 .96132 Mdot
.77895 .15468 Mdot
.75457 .14297 Mdot
.80768 .17247 Mdot
.45121 .97577 Mdot
.49721 .96988 Mdot
.8482 .79433 Mdot
.44237 .02392 Mdot
.36466 .03071 Mdot
.49184 .02903 Mdot
.78264 .84334 Mdot
.81254 .82404 Mdot
.97209 .55625 Mdot
.09649 .22655 Mdot
.02425 .52421 Mdot
.24536 .07615 Mdot
.89102 .74657 Mdot
.9376 .66999 Mdot
.56174 .90161 Mdot
.65162 .13531 Mdot
.64427 .13906 Mdot
.65935 .1322 Mdot
.56982 .91177 Mdot
.56639 .90663 Mdot
.53942 .88598 Mdot
.63097 .14922 Mdot
.62434 .15563 Mdot
.63747 .14368 Mdot
.55558 .89664 Mdot
.54809 .89151 Mdot
.56634 .9364 Mdot
.71678 .13122 Mdot
.70182 .12886 Mdot
.73433 .13576 Mdot
.54622 .9525 Mdot
.55886 .94407 Mdot
.57197 .91715 Mdot
.67783 .12853 Mdot
.66791 .12987 Mdot
.6891 .12816 Mdot
.57053 .9294 Mdot
.5724 .92297 Mdot
.49406 .14325 Mdot
.59322 .80948 Mdot
.50754 .13396 Mdot
.6026 .82027 Mdot
.58284 .79722 Mdot
.46399 .16713 Mdot
.47947 .15422 Mdot
.62514 .84546 Mdot
.5311 .11923 Mdot
.54095 .11323 Mdot
.51992 .12605 Mdot
.6109 .82979 Mdot
.61828 .83817 Mdot
.40328 .24021 Mdot
.51355 .7159 Mdot
.5289 .73444 Mdot
.49818 .69678 Mdot
.38083 .28625 Mdot
.3911 .26285 Mdot
.44809 .18219 Mdot
.55813 .76845 Mdot
.57119 .78353 Mdot
.54386 .75206 Mdot
.41716 .21893 Mdot
.43228 .19949 Mdot
.70405 .8708 Mdot
.57028 .06967 Mdot
.56548 .06253 Mdot
.57213 .0761 Mdot
.67929 .87165 Mdot
.6907 .87191 Mdot
.75804 .8555 Mdot
.54412 .04628 Mdot
.52331 .03737 Mdot
.55718 .05471 Mdot
.71917 .86827 Mdot
.73695 .8635 Mdot
.64534 .86153 Mdot
.5564 .10259 Mdot
.56236 .09763 Mdot
.54937 .10775 Mdot
.63191 .85166 Mdot
.63862 .85696 Mdot
.66945 .8703 Mdot
.5704 .08752 Mdot
.57199 .08199 Mdot
.56711 .09267 Mdot
.65251 .8653 Mdot
.66059 .86817 Mdot
.36168 .35739 Mdot
.4451 .6192 Mdot
.36585 .33548 Mdot
.45582 .63718 Mdot
.43588 .60212 Mdot
.35727 .39721 Mdot
.35893 .37802 Mdot
.49636 .69448 Mdot
.37965 .2891 Mdot
.38969 .26563 Mdot
.37174 .31258 Mdot
.46809 .65591 Mdot
.48176 .67511 Mdot
.35633 .4607 Mdot
.41011 .53043 Mdot
.41291 .54322 Mdot
.40829 .51814 Mdot
.35666 .48716 Mdot
.35651 .47418 Mdot
.35643 .41496 Mdot
.42181 .57087 Mdot
.42811 .586 Mdot
.41679 .55666 Mdot
.35617 .44648 Mdot
.35614 .43132 Mdot
.60979 .8288 Mdot
.50613 .135 Mdot
.51869 .12695 Mdot
.49232 .14446 Mdot
.59221 .80803 Mdot
.6015 .81907 Mdot
.63111 .85097 Mdot
.53966 .11387 Mdot
.54835 .10836 Mdot
.52981 .11999 Mdot
.61745 .83723 Mdot
.62452 .84456 Mdot
.54223 .74998 Mdot
.41554 .22136 Mdot
.43056 .20168 Mdot
.40176 .24285 Mdot
.5116 .71367 Mdot
.52712 .73227 Mdot
.58157 .79565 Mdot
.462 .1688 Mdot
.4775 .15564 Mdot
.44623 .18413 Mdot
.55644 .76659 Mdot
.56956 .78186 Mdot
.41513 .77809 Mdot
.54184 .25049 Mdot
.52666 .26821 Mdot
.5561 .23386 Mdot
.44584 .81544 Mdot
.43015 .79782 Mdot
.37941 .71027 Mdot
.49598 .30605 Mdot
.48137 .32542 Mdot
.51119 .28684 Mdot
.40139 .75656 Mdot
.38938 .73374 Mdot
.50574 .86477 Mdot
.60959 .17146 Mdot
.6013 .18123 Mdot
.61719 .16296 Mdot
.52959 .87983 Mdot
.51836 .87285 Mdot
.46161 .83082 Mdot
.58121 .20469 Mdot
.56923 .21854 Mdot
.59195 .1923 Mdot
.4919 .85527 Mdot
.47709 .84404 Mdot
.35633 .53893 Mdot
.41282 .45715 Mdot
.41005 .46991 Mdot
.41667 .44372 Mdot
.35615 .56828 Mdot
.35617 .55313 Mdot
.35671 .49984 Mdot
.40737 .4942 Mdot
.40739 .50613 Mdot
.40826 .48219 Mdot
.35651 .52546 Mdot
.35666 .5125 Mdot
.36159 .64204 Mdot
.45552 .36332 Mdot
.44484 .38128 Mdot
.46773 .34461 Mdot
.37158 .68681 Mdot
.36573 .66393 Mdot
.35642 .58459 Mdot
.42792 .41443 Mdot
.42166 .42953 Mdot
.43564 .39833 Mdot
.35886 .62144 Mdot
.35724 .60229 Mdot
.35669 .49166 Mdot
.40728 .50185 Mdot
.35657 .47884 Mdot
.40786 .51381 Mdot
.40759 .48995 Mdot
.35662 .51713 Mdot
.35671 .50435 Mdot
.41526 .55175 Mdot
.3562 .4517 Mdot
.35613 .43686 Mdot
.35638 .46562 Mdot
.40936 .52601 Mdot
.4118 .53859 Mdot
.3562 .57396 Mdot
.42375 .42424 Mdot
.41829 .43876 Mdot
.43054 .40878 Mdot
.35774 .609 Mdot
.35667 .5908 Mdot
.35645 .53023 Mdot
.41093 .46539 Mdot
.40878 .47786 Mdot
.41407 .45243 Mdot
.35612 .55838 Mdot
.35625 .54388 Mdot
.5062 .70684 Mdot
.38594 .27395 Mdot
.39721 .25087 Mdot
.37667 .29748 Mdot
.47668 .66821 Mdot
.49102 .68757 Mdot
.55149 .76079 Mdot
.42497 .2085 Mdot
.44054 .19015 Mdot
.41035 .22887 Mdot
.52159 .72571 Mdot
.53679 .74381 Mdot
.43292 .59624 Mdot
.35692 .40368 Mdot
.3582 .38507 Mdot
.35631 .42091 Mdot
.41987 .56569 Mdot
.42574 .5805 Mdot
.46358 .64914 Mdot
.36414 .34347 Mdot
.36945 .32084 Mdot
.3605 .36496 Mdot
.44158 .61298 Mdot
.45185 .63066 Mdot
.62657 .15315 Mdot
.54283 .88798 Mdot
.61981 .16023 Mdot
.53323 .88211 Mdot
.55103 .89335 Mdot
.64005 .14202 Mdot
.63325 .14712 Mdot
.49707 .85884 Mdot
.60437 .17759 Mdot
.59528 .18812 Mdot
.61255 .16836 Mdot
.52236 .87549 Mdot
.51029 .86782 Mdot
.67148 .12937 Mdot
.57219 .91922 Mdot
.57092 .91361 Mdot
.57187 .92523 Mdot
.6933 .12817 Mdot
.6816 .12823 Mdot
.64686 .13764 Mdot
.56341 .90343 Mdot
.55778 .89847 Mdot
.56788 .90841 Mdot
.66231 .13128 Mdot
.65412 .13402 Mdot
.37416 .69517 Mdot
.48642 .31849 Mdot
.47249 .33779 Mdot
.50136 .29914 Mdot
.39354 .74203 Mdot
.38282 .71873 Mdot
.35967 .62863 Mdot
.44844 .37496 Mdot
.43873 .39235 Mdot
.45977 .35671 Mdot
.36756 .67198 Mdot
.36283 .64968 Mdot
.45142 .82119 Mdot
.57379 .21348 Mdot
.56095 .22824 Mdot
.58518 .20009 Mdot
.48263 .84826 Mdot
.46727 .83578 Mdot
.40615 .76444 Mdot
.53214 .26176 Mdot
.51682 .28013 Mdot
.54694 .24437 Mdot
.43557 .80436 Mdot
.42029 .78535 Mdot
.74361 .1388 Mdot
.53804 .95668 Mdot
.72483 .13309 Mdot
.55364 .94799 Mdot
.51454 .96551 Mdot
.79206 .16226 Mdot
.76556 .14783 Mdot
.57187 .92594 Mdot
.69498 .12835 Mdot
.68302 .12822 Mdot
.70863 .12974 Mdot
.56323 .93997 Mdot
.56885 .93266 Mdot
.95168 .36367 Mdot
.05628 .68826 Mdot
.19666 .89011 Mdot
.03911 .36908 Mdot
.95893 .6148 Mdot
.97598 .4871 Mdot
.82344 .18419 Mdot
.42061 .97592 Mdot
.47772 .97328 Mdot
.3304 .96087 Mdot
.90653 .27514 Mdot
.86169 .21914 Mdot
.5239 .87629 Mdot
.6206 .15931 Mdot
.61327 .16725 Mdot
.62756 .15247 Mdot
.5437 .88868 Mdot
.5345 .88284 Mdot
.48424 .84962 Mdot
.59658 .18687 Mdot
.58646 .19859 Mdot
.6054 .17643 Mdot
.51188 .86878 Mdot
.4986 .86 Mdot
.5682 .90905 Mdot
.66334 .13101 Mdot
.65525 .13375 Mdot
.67242 .12908 Mdot
.57238 .91987 Mdot
.57098 .9143 Mdot
.55177 .894 Mdot
.64065 .14137 Mdot
.6341 .14649 Mdot
.64766 .13715 Mdot
.56419 .90397 Mdot
.55865 .89902 Mdot
.63546 .85452 Mdot
.54551 .1103 Mdot
.55314 .10501 Mdot
.53645 .11601 Mdot
.62189 .84213 Mdot
.62872 .84881 Mdot
.65667 .86689 Mdot
.56505 .09506 Mdot
.56903 .09001 Mdot
.55969 .09999 Mdot
.64207 .85947 Mdot
.64898 .86362 Mdot
.58841 .80383 Mdot
.47219 .16011 Mdot
.4872 .14825 Mdot
.45647 .17402 Mdot
.56514 .77651 Mdot
.57742 .79089 Mdot
.61482 .83434 Mdot
.51418 .12966 Mdot
.52597 .12237 Mdot
.50121 .13819 Mdot
.5983 .81529 Mdot
.60709 .8254 Mdot
.79768 .83422 Mdot
.50857 .03284 Mdot
.46861 .02561 Mdot
.53429 .04162 Mdot
.74766 .8597 Mdot
.77033 .84985 Mdot
.91565 .71062 Mdot
.30804 .04654 Mdot
.16717 .13618 Mdot
.4063 .02493 Mdot
.83028 .81026 Mdot
.86992 .77198 Mdot
.68503 .87195 Mdot
.5723 .07897 Mdot
.57151 .07282 Mdot
.57135 .08465 Mdot
.66513 .86937 Mdot
.6744 .87114 Mdot
.72807 .86613 Mdot
.56161 .05851 Mdot
.55114 .05038 Mdot
.56816 .06602 Mdot
.69753 .87146 Mdot
.71165 .86973 Mdot
.49957 .30144 Mdot
.39206 .73926 Mdot
.48478 .32081 Mdot
.3816 .71589 Mdot
.40451 .76182 Mdot
.53032 .2639 Mdot
.51487 .28234 Mdot
.36243 .64714 Mdot
.4583 .35892 Mdot
.44725 .37707 Mdot
.47085 .34007 Mdot
.37327 .69238 Mdot
.36697 .66931 Mdot
.58387 .2016 Mdot
.48069 .84689 Mdot
.46536 .83416 Mdot
.49528 .85768 Mdot
.60338 .17879 Mdot
.5943 .18954 Mdot
.54534 .24642 Mdot
.43384 .80222 Mdot
.4186 .78296 Mdot
.44963 .81931 Mdot
.57219 .2151 Mdot
.55935 .23009 Mdot
.35671 .50286 Mdot
.4075 .49135 Mdot
.4073 .50329 Mdot
.4086 .4793 Mdot
.35647 .52861 Mdot
.35663 .51556 Mdot
.35637 .46397 Mdot
.40959 .52746 Mdot
.41215 .54012 Mdot
.408 .51526 Mdot
.35668 .4902 Mdot
.35655 .47731 Mdot
.35655 .58868 Mdot
.42962 .41068 Mdot
.42304 .42601 Mdot
.43769 .39436 Mdot
.35939 .62622 Mdot
.35753 .60672 Mdot
.35628 .54223 Mdot
.41363 .45401 Mdot
.41062 .46692 Mdot
.41776 .44041 Mdot
.35618 .57206 Mdot
.35615 .55665 Mdot
.48795 .97167 Mdot
.81537 .178 Mdot
.52112 .96343 Mdot
.78537 .15829 Mdot
.85201 .20938 Mdot
.35576 .96744 Mdot
.43679 .97619 Mdot
.72073 .13209 Mdot
.5564 .94601 Mdot
.56489 .93816 Mdot
.54234 .95458 Mdot
.75995 .14527 Mdot
.73889 .13719 Mdot
.13086 .17658 Mdot
.92677 .69137 Mdot
.96644 .58657 Mdot
.8803 .76 Mdot
.38655 .02706 Mdot
.27786 .05908 Mdot
.23205 .91568 Mdot
.94179 .33921 Mdot
.89542 .25932 Mdot
.97363 .4555 Mdot
.02637 .44538 Mdot
.08415 .75145 Mdot
.63254 .14783 Mdot
.54997 .89276 Mdot
.54159 .88735 Mdot
.55715 .89784 Mdot
.64595 .13808 Mdot
.63905 .1425 Mdot
.61145 .16932 Mdot
.52117 .8746 Mdot
.50885 .86682 Mdot
.53208 .88136 Mdot
.62596 .15402 Mdot
.61891 .16111 Mdot
.68039 .12835 Mdot
.5722 .92444 Mdot
.57222 .9185 Mdot
.56976 .93101 Mdot
.70516 .12925 Mdot
.69199 .12822 Mdot
.65342 .13451 Mdot
.56733 .90784 Mdot
.563 .90279 Mdot
.57044 .91303 Mdot
.67013 .12945 Mdot
.66132 .13158 Mdot
.53186 .04057 Mdot
.74996 .85889 Mdot
.50462 .03183 Mdot
.77349 .8482 Mdot
.73041 .86541 Mdot
.56082 .05763 Mdot
.54932 .04933 Mdot
.87497 .76629 Mdot
.39664 .02585 Mdot
.29334 .05228 Mdot
.46275 .02503 Mdot
.80112 .83201 Mdot
.83479 .80641 Mdot
.5717 .08404 Mdot
.66598 .86973 Mdot
.6757 .87123 Mdot
.6576 .86724 Mdot
.56545 .09443 Mdot
.56922 .08935 Mdot
.56752 .06516 Mdot
.69892 .87145 Mdot
.71348 .8694 Mdot
.68658 .87187 Mdot
.57246 .0783 Mdot
.57105 .072 Mdot
.76294 .14668 Mdot
.51772 .9645 Mdot
.54047 .95559 Mdot
.48319 .97249 Mdot
.81956 .18121 Mdot
.78847 .16004 Mdot
.70709 .12961 Mdot
.56413 .93905 Mdot
.56953 .93178 Mdot
.55484 .94704 Mdot
.74107 .13787 Mdot
.72273 .13256 Mdot
.97513 .4714 Mdot
.03148 .40664 Mdot
.06945 .72106 Mdot
.14884 .15529 Mdot
.92113 .70144 Mdot
.96295 .60071 Mdot
.85669 .21399 Mdot
.34328 .96443 Mdot
.4288 .97614 Mdot
.21466 .90386 Mdot
.94678 .35097 Mdot
.9008 .26677 Mdot
.37758 .29472 Mdot
.47841 .6705 Mdot
.4928 .68987 Mdot
.46504 .65139 Mdot
.36471 .34081 Mdot
.37015 .31811 Mdot
.41214 .22633 Mdot
.52344 .72791 Mdot
.5387 .74585 Mdot
.50793 .70914 Mdot
.38712 .27119 Mdot
.39873 .24818 Mdot
.35632 .41898 Mdot
.4205 .5674 Mdot
.42648 .58231 Mdot
.41577 .55339 Mdot
.3562 .44995 Mdot
.35612 .43504 Mdot
.36091 .36243 Mdot
.44276 .61505 Mdot
.45312 .63282 Mdot
.4339 .59819 Mdot
.35701 .40157 Mdot
.35845 .38272 Mdot
.53741 .11526 Mdot
.6229 .8429 Mdot
.62954 .84953 Mdot
.61569 .83532 Mdot
.51582 .12876 Mdot
.52725 .12156 Mdot
.56054 .09944 Mdot
.64286 .86001 Mdot
.65006 .86395 Mdot
.63605 .85524 Mdot
.5464 .10964 Mdot
.55416 .10444 Mdot
.45824 .17225 Mdot
.56654 .77835 Mdot
.57884 .79249 Mdot
.55313 .76275 Mdot
.4269 .20619 Mdot
.44245 .18811 Mdot
.50296 .13712 Mdot
.59937 .81657 Mdot
.60788 .8266 Mdot
.5898 .8052 Mdot
.47386 .15859 Mdot
.48889 .14696 Mdot
.35613 .43938 Mdot
.41462 .54953 Mdot
.35623 .42368 Mdot
.41904 .56333 Mdot
.41131 .53645 Mdot
.35642 .46785 Mdot
.35624 .45403 Mdot
.44009 .61014 Mdot
.35793 .38826 Mdot
.36007 .36837 Mdot
.35674 .40668 Mdot
.42465 .57796 Mdot
.43161 .59355 Mdot
.35659 .51924 Mdot
.40908 .47582 Mdot
.40773 .48793 Mdot
.41137 .4633 Mdot
.35624 .54626 Mdot
.35642 .53244 Mdot
.35659 .48103 Mdot
.40771 .51183 Mdot
.40904 .52395 Mdot
.40727 .49988 Mdot
.3567 .50644 Mdot
.3567 .49381 Mdot
.56286 .77403 Mdot
.438 .19304 Mdot
.45381 .17655 Mdot
.42257 .21174 Mdot
.5344 .74084 Mdot
.54916 .75804 Mdot
.59681 .81345 Mdot
.48465 .15008 Mdot
.499 .13976 Mdot
.46946 .16227 Mdot
.5754 .78863 Mdot
.5868 .80173 Mdot
.47444 .66502 Mdot
.36845 .32465 Mdot
.37526 .30142 Mdot
.36348 .34709 Mdot
.45003 .62767 Mdot
.46148 .64603 Mdot
.51897 .72264 Mdot
.39518 .25466 Mdot
.40811 .23242 Mdot
.38417 .27788 Mdot
.48863 .68434 Mdot
.50359 .70367 Mdot
.59697 .18632 Mdot
.49932 .86043 Mdot
.58704 .19804 Mdot
.48497 .85015 Mdot
.51253 .86916 Mdot
.61365 .16686 Mdot
.60572 .17594 Mdot
.4383 .80732 Mdot
.56309 .22566 Mdot
.54941 .24162 Mdot
.57567 .21111 Mdot
.46976 .838 Mdot
.45406 .82377 Mdot
.64098 .14116 Mdot
.55899 .89925 Mdot
.55221 .89422 Mdot
.56429 .90424 Mdot
.65556 .13358 Mdot
.64813 .13703 Mdot
.62105 .15902 Mdot
.53484 .88315 Mdot
.52439 .87663 Mdot
.54412 .88893 Mdot
.63432 .14619 Mdot
.62782 .15215 Mdot
.35798 .61214 Mdot
.44027 .38952 Mdot
.43178 .40612 Mdot
.45023 .37198 Mdot
.36356 .65335 Mdot
.36014 .63206 Mdot
.35613 .56092 Mdot
.41913 .4364 Mdot
.41471 .4502 Mdot
.42477 .42174 Mdot
.35677 .59368 Mdot
.35623 .57665 Mdot
.39545 .74581 Mdot
.5193 .277 Mdot
.50387 .29594 Mdot
.53472 .25882 Mdot
.42288 .78867 Mdot
.40842 .76803 Mdot
.36853 .67579 Mdot
.47474 .33461 Mdot
.46172 .3536 Mdot
.48891 .31528 Mdot
.38438 .7226 Mdot
.3754 .69905 Mdot
.68472 .12806 Mdot
.57152 .92702 Mdot
.67418 .12892 Mdot
.57227 .92089 Mdot
.56834 .93378 Mdot
.71128 .13024 Mdot
.69714 .1285 Mdot
.56489 .90482 Mdot
.65638 .13316 Mdot
.64881 .1365 Mdot
.6649 .1307 Mdot
.57136 .91519 Mdot
.56895 .90986 Mdot
.7968 .1652 Mdot
.47001 .97425 Mdot
.50936 .96695 Mdot
.4084 .97523 Mdot
.86879 .22678 Mdot
.82936 .18899 Mdot
.72759 .13375 Mdot
.55145 .9494 Mdot
.56182 .94128 Mdot
.53494 .95813 Mdot
.7697 .14986 Mdot
.74708 .14008 Mdot
.47177 .83955 Mdot
.58814 .19651 Mdot
.57714 .2095 Mdot
.59799 .18499 Mdot
.50088 .86156 Mdot
.48685 .85146 Mdot
.42455 .79098 Mdot
.55106 .23964 Mdot
.5364 .25668 Mdot
.56479 .2239 Mdot
.45601 .82558 Mdot
.44009 .80939 Mdot
.54532 .88955 Mdot
.63527 .14562 Mdot
.62849 .15134 Mdot
.64193 .14068 Mdot
.5595 .89989 Mdot
.55299 .89485 Mdot
.51383 .87013 Mdot
.61465 .16591 Mdot
.60686 .17486 Mdot
.62172 .15808 Mdot
.53616 .88384 Mdot
.52563 .87746 Mdot
.66353 .86908 Mdot
.56826 .09081 Mdot
.57108 .08557 Mdot
.56429 .09589 Mdot
.64793 .86292 Mdot
.65544 .86634 Mdot
.69528 .87166 Mdot
.57177 .07388 Mdot
.56874 .06716 Mdot
.57243 .07999 Mdot
.67269 .87096 Mdot
.68336 .87177 Mdot
.62086 .84086 Mdot
.52421 .12352 Mdot
.53469 .11699 Mdot
.51225 .13101 Mdot
.60559 .82386 Mdot
.61347 .83298 Mdot
.64083 .85877 Mdot
.55201 .10588 Mdot
.55885 .10086 Mdot
.54393 .11118 Mdot
.62773 .84771 Mdot
.63424 .85368 Mdot
.95272 .6335 Mdot
.19261 .1132 Mdot
.05357 .31942 Mdot
.32731 .04008 Mdot
.86276 .77975 Mdot
.90776 .72302 Mdot
.91438 .28732 Mdot
.17126 .86779 Mdot
.31131 .95464 Mdot
.04116 .63919 Mdot
.97608 .50918 Mdot
.95799 .3822 Mdot
.74411 .86104 Mdot
.55326 .05178 Mdot
.53755 .04309 Mdot
.56308 .05984 Mdot
.70902 .87022 Mdot
.72535 .86676 Mdot
.8244 .81505 Mdot
.47652 .02657 Mdot
.41859 .02417 Mdot
.51377 .03427 Mdot
.76624 .85184 Mdot
.79281 .83729 Mdot
.44897 .37411 Mdot
.36302 .65071 Mdot
.43913 .39156 Mdot
.35977 .62957 Mdot
.36786 .67307 Mdot
.47308 .33689 Mdot
.46033 .35583 Mdot
.35623 .57474 Mdot
.42406 .42352 Mdot
.41852 .4381 Mdot
.43088 .40803 Mdot
.35778 .60987 Mdot
.35671 .59162 Mdot
.53281 .2609 Mdot
.42097 .78629 Mdot
.4067 .76544 Mdot
.43634 .8052 Mdot
.56165 .22754 Mdot
.54764 .24359 Mdot
.4871 .31758 Mdot
.38328 .71984 Mdot
.37456 .69629 Mdot
.39405 .7431 Mdot
.51752 .27924 Mdot
.50214 .29825 Mdot
.3562 .45238 Mdot
.41167 .53799 Mdot
.41508 .55113 Mdot
.40926 .52542 Mdot
.35658 .47944 Mdot
.3564 .46623 Mdot
.35689 .40452 Mdot
.42543 .57978 Mdot
.43257 .59548 Mdot
.41961 .56501 Mdot
.35611 .4376 Mdot
.35628 .4217 Mdot
.35644 .53082 Mdot
.41106 .4648 Mdot
.40887 .47726 Mdot
.41424 .45181 Mdot
.35613 .5591 Mdot
.35624 .54453 Mdot
.3567 .49228 Mdot
.40728 .5013 Mdot
.40781 .51324 Mdot
.40762 .48939 Mdot
.35661 .51774 Mdot
.35671 .50497 Mdot
.56366 .93944 Mdot
.72344 .13269 Mdot
.56937 .9321 Mdot
.70773 .12968 Mdot
.74217 .1383 Mdot
.5396 .95601 Mdot
.55439 .94741 Mdot
.67185 .12926 Mdot
.57219 .91949 Mdot
.57091 .9139 Mdot
.57191 .9255 Mdot
.69404 .12829 Mdot
.68205 .12816 Mdot
.33836 .96313 Mdot
.90305 .27001 Mdot
.85876 .21612 Mdot
.94882 .3561 Mdot
.064 .70826 Mdot
.20753 .89861 Mdot
.51643 .9649 Mdot
.78983 .16086 Mdot
.76397 .14713 Mdot
.82108 .18237 Mdot
.42571 .97609 Mdot
.48113 .97281 Mdot
.60487 .17717 Mdot
.51085 .86821 Mdot
.49761 .85931 Mdot
.52295 .87581 Mdot
.62006 .15984 Mdot
.61286 .16793 Mdot
.57431 .2128 Mdot
.46807 .83642 Mdot
.45223 .82192 Mdot
.48333 .8488 Mdot
.59575 .1876 Mdot
.58562 .19947 Mdot
.6471 .13741 Mdot
.56374 .90364 Mdot
.55805 .8987 Mdot
.56812 .90863 Mdot
.66283 .13123 Mdot
.65451 .13388 Mdot
.62689 .15285 Mdot
.54326 .88824 Mdot
.5338 .88239 Mdot
.55125 .89363 Mdot
.64032 .14178 Mdot
.63369 .14692 Mdot
.3137 .04453 Mdot
.86793 .77418 Mdot
.17431 .12933 Mdot
.91339 .71428 Mdot
.8287 .81155 Mdot
.47108 .02587 Mdot
.40991 .02467 Mdot
.95348 .36865 Mdot
.05159 .67476 Mdot
.18959 .88427 Mdot
.04276 .35474 Mdot
.95729 .62003 Mdot
.97614 .49354 Mdot
.56201 .05888 Mdot
.71102 .86979 Mdot
.72726 .86633 Mdot
.69678 .87158 Mdot
.5715 .0731 Mdot
.56845 .06637 Mdot
.50994 .03322 Mdot
.76927 .85034 Mdot
.79613 .83525 Mdot
.74658 .86014 Mdot
.55165 .05076 Mdot
.53541 .04206 Mdot
.69548 .12833 Mdot
.56869 .93298 Mdot
.57163 .92628 Mdot
.56298 .94031 Mdot
.72572 .13335 Mdot
.70934 .12985 Mdot
.66366 .13086 Mdot
.57117 .91453 Mdot
.56833 .9093 Mdot
.57246 .92013 Mdot
.68359 .12823 Mdot
.67295 .12905 Mdot
.82518 .1856 Mdot
.4171 .97576 Mdot
.47565 .97356 Mdot
.32501 .95921 Mdot
.90867 .27835 Mdot
.86356 .2211 Mdot
.74446 .13907 Mdot
.53721 .95709 Mdot
.55291 .94841 Mdot
.51315 .96592 Mdot
.79334 .16303 Mdot
.7668 .14846 Mdot
.42425 .20941 Mdot
.53612 .74297 Mdot
.55074 .76005 Mdot
.52091 .72482 Mdot
.39671 .25192 Mdot
.40974 .22986 Mdot
.47143 .16072 Mdot
.57694 .79022 Mdot
.58796 .80323 Mdot
.56451 .77581 Mdot
.43974 .19097 Mdot
.45566 .17473 Mdot
.36392 .34452 Mdot
.4513 .62981 Mdot
.46301 .64826 Mdot
.44114 .61217 Mdot
.35815 .38594 Mdot
.36037 .36593 Mdot
.38547 .27505 Mdot
.49029 .68666 Mdot
.50546 .70595 Mdot
.47606 .66731 Mdot
.36913 .32194 Mdot
.37626 .2986 Mdot
.56474 .09527 Mdot
.64871 .86341 Mdot
.65619 .8668 Mdot
.64183 .85922 Mdot
.55293 .10528 Mdot
.55937 .10021 Mdot
.57226 .07924 Mdot
.67403 .87103 Mdot
.68451 .87193 Mdot
.6647 .86928 Mdot
.56886 .09025 Mdot
.57142 .08495 Mdot
.51359 .13003 Mdot
.60667 .82497 Mdot
.61454 .83391 Mdot
.59777 .81482 Mdot
.48657 .14877 Mdot
.50068 .13864 Mdot
.54515 .11057 Mdot
.62833 .84856 Mdot
.63509 .8543 Mdot
.62162 .84176 Mdot
.52537 .12267 Mdot
.53592 .11628 Mdot
.57212 .07539 Mdot
.68073 .87165 Mdot
.56966 .06881 Mdot
.69228 .8718 Mdot
.67039 .8706 Mdot
.57054 .08684 Mdot
.57229 .08136 Mdot
.73937 .86267 Mdot
.55608 .05377 Mdot
.54188 .04519 Mdot
.56476 .06165 Mdot
.70553 .87072 Mdot
.72126 .86776 Mdot
.55021 .10711 Mdot
.63267 .85235 Mdot
.63922 .85764 Mdot
.62613 .84616 Mdot
.53229 .11847 Mdot
.54183 .11251 Mdot
.5674 .09202 Mdot
.65361 .86559 Mdot
.6615 .86851 Mdot
.64622 .86199 Mdot
.55736 .10204 Mdot
.56311 .09707 Mdot
.92555 .30641 Mdot
.13495 .82852 Mdot
.28139 .94255 Mdot
.02729 .56341 Mdot
.97402 .54102 Mdot
.96568 .41016 Mdot
.80409 .17002 Mdot
.45766 .97537 Mdot
.50154 .96892 Mdot
.38897 .97326 Mdot
.87914 .23863 Mdot
.83803 .19638 Mdot
.81625 .82135 Mdot
.48685 .02814 Mdot
.43502 .02381 Mdot
.52044 .03635 Mdot
.76058 .85446 Mdot
.78611 .8413 Mdot
.94294 .65817 Mdot
.22811 .08689 Mdot
.08068 .25526 Mdot
.35303 .03318 Mdot
.85305 .78962 Mdot
.89667 .73898 Mdot
.42982 .58975 Mdot
.35657 .41086 Mdot
.43794 .6061 Mdot
.35758 .39278 Mdot
.35617 .42753 Mdot
.41788 .55997 Mdot
.42319 .57439 Mdot
.3671 .33009 Mdot
.45861 .64159 Mdot
.47123 .66045 Mdot
.44752 .62341 Mdot
.35947 .37323 Mdot
.36254 .35228 Mdot
.40752 .50898 Mdot
.35671 .49683 Mdot
.35663 .4841 Mdot
.35668 .50947 Mdot
.40797 .48507 Mdot
.4073 .49704 Mdot
.41374 .54636 Mdot
.35628 .45739 Mdot
.35615 .44296 Mdot
.35647 .47103 Mdot
.40864 .52103 Mdot
.41068 .53342 Mdot
.4811 .15281 Mdot
.58422 .79873 Mdot
.59454 .81078 Mdot
.57252 .78529 Mdot
.45002 .18027 Mdot
.46575 .16547 Mdot
.52148 .12521 Mdot
.61165 .83093 Mdot
.61917 .83907 Mdot
.60357 .8215 Mdot
.49569 .14206 Mdot
.50923 .13297 Mdot
.39238 .26012 Mdot
.49996 .69909 Mdot
.51529 .71817 Mdot
.48517 .67972 Mdot
.37344 .30701 Mdot
.38185 .28348 Mdot
.43427 .1973 Mdot
.54572 .75404 Mdot
.55968 .77035 Mdot
.53077 .73658 Mdot
.40489 .23759 Mdot
.41902 .21649 Mdot
.41613 .44537 Mdot
.35613 .5664 Mdot
.41245 .45868 Mdot
.35616 .55135 Mdot
.35639 .58262 Mdot
.42714 .41626 Mdot
.42099 .43127 Mdot
.35668 .51103 Mdot
.4081 .48365 Mdot
.40733 .49563 Mdot
.40979 .47138 Mdot
.35634 .53727 Mdot
.35654 .52393 Mdot
.46625 .34687 Mdot
.37079 .68405 Mdot
.3651 .66122 Mdot
.37845 .7075 Mdot
.49415 .30834 Mdot
.47962 .32771 Mdot
.43464 .40031 Mdot
.35861 .61908 Mdot
.35714 .60015 Mdot
.36114 .63947 Mdot
.4542 .3655 Mdot
.44361 .38336 Mdot
.35915 .37566 Mdot
.43686 .6041 Mdot
.44626 .6213 Mdot
.42899 .58788 Mdot
.35653 .41287 Mdot
.35744 .39498 Mdot
.37263 .30978 Mdot
.46967 .65817 Mdot
.48337 .67742 Mdot
.45726 .63938 Mdot
.36204 .35489 Mdot
.36643 .33282 Mdot
.35649 .47259 Mdot
.40845 .51958 Mdot
.4104 .53194 Mdot
.40745 .50753 Mdot
.35671 .49832 Mdot
.35665 .48558 Mdot
.35617 .42942 Mdot
.4173 .55829 Mdot
.4225 .57263 Mdot
.41332 .54478 Mdot
.35629 .45909 Mdot
.35614 .44477 Mdot
.65831 .13249 Mdot
.56969 .91109 Mdot
.56596 .90601 Mdot
.57173 .91651 Mdot
.67657 .12863 Mdot
.66703 .13021 Mdot
.63686 .14438 Mdot
.55466 .89607 Mdot
.5473 .89084 Mdot
.56088 .90107 Mdot
.65054 .13563 Mdot
.64355 .13962 Mdot
.7319 .13495 Mdot
.54818 .95144 Mdot
.55975 .94316 Mdot
.52975 .96027 Mdot
.77569 .1529 Mdot
.75217 .14206 Mdot
.68748 .12803 Mdot
.571 .92857 Mdot
.57225 .9223 Mdot
.56714 .9355 Mdot
.71496 .1309 Mdot
.70035 .12878 Mdot
.55442 .23576 Mdot
.44385 .81345 Mdot
.42818 .79556 Mdot
.45979 .8291 Mdot
.57989 .20628 Mdot
.56787 .22035 Mdot
.50946 .28913 Mdot
.3999 .75391 Mdot
.38818 .731 Mdot
.41334 .7756 Mdot
.53995 .25251 Mdot
.52486 .27041 Mdot
.61641 .16395 Mdot
.52824 .87906 Mdot
.51675 .87198 Mdot
.53846 .88525 Mdot
.63008 .14988 Mdot
.62336 .15638 Mdot
.59059 .19364 Mdot
.49031 .85402 Mdot
.47543 .84257 Mdot
.50408 .86373 Mdot
.60881 .17263 Mdot
.60018 .18244 Mdot
.5741 .78691 Mdot
.45187 .1784 Mdot
.58544 .80027 Mdot
.46773 .16386 Mdot
.436 .19517 Mdot
.54733 .75609 Mdot
.56136 .77217 Mdot
.5106 .13196 Mdot
.60466 .82266 Mdot
.61274 .83188 Mdot
.59553 .81219 Mdot
.48304 .15144 Mdot
.4974 .1409 Mdot
.4868 .68204 Mdot
.37438 .3042 Mdot
.38309 .28064 Mdot
.36773 .3274 Mdot
.4601 .6438 Mdot
.47283 .66273 Mdot
.53253 .73873 Mdot
.40647 .23499 Mdot
.42067 .21412 Mdot
.39385 .25736 Mdot
.5018 .70138 Mdot
.51723 .72038 Mdot
.57217 .08063 Mdot
.67171 .87069 Mdot
.68186 .87182 Mdot
.66262 .86875 Mdot
.56802 .09147 Mdot
.57095 .08624 Mdot
.56385 .06073 Mdot
.70747 .87034 Mdot
.72314 .86737 Mdot
.69368 .87179 Mdot
.57188 .07462 Mdot
.56945 .06804 Mdot
.54309 .11188 Mdot
.62674 .84703 Mdot
.6335 .853 Mdot
.61996 .84 Mdot
.52268 .12433 Mdot
.53355 .11773 Mdot
.56358 .09645 Mdot
.64701 .86249 Mdot
.65432 .86607 Mdot
.64022 .85811 Mdot
.55119 .10651 Mdot
.55792 .1014 Mdot
.3569 .59585 Mdot
.43272 .4042 Mdot
.3563 .57864 Mdot
.42556 .41992 Mdot
.44133 .38749 Mdot
.36042 .63448 Mdot
.35817 .61443 Mdot
.41172 .46176 Mdot
.3562 .54791 Mdot
.35639 .53403 Mdot
.35612 .56272 Mdot
.41972 .43471 Mdot
.41515 .44861 Mdot
.38567 .72543 Mdot
.50579 .29368 Mdot
.4906 .31296 Mdot
.5212 .2748 Mdot
.40998 .77056 Mdot
.39692 .74853 Mdot
.36401 .65593 Mdot
.46325 .35137 Mdot
.45154 .36983 Mdot
.47632 .33231 Mdot
.37644 .70189 Mdot
.36927 .67854 Mdot
.42393 .57618 Mdot
.35621 .4256 Mdot
.35669 .40873 Mdot
.35612 .44122 Mdot
.41417 .54793 Mdot
.41842 .56162 Mdot
.44873 .62553 Mdot
.35973 .37084 Mdot
.36293 .34974 Mdot
.35777 .3905 Mdot
.43074 .59165 Mdot
.43896 .60811 Mdot
.40728 .49847 Mdot
.35669 .50799 Mdot
.35671 .4953 Mdot
.35658 .52083 Mdot
.4093 .47433 Mdot
.40783 .48653 Mdot
.41101 .53495 Mdot
.35645 .46943 Mdot
.35624 .45576 Mdot
.35662 .48252 Mdot
.4076 .51038 Mdot
.40883 .52249 Mdot
.45784 .82736 Mdot
.57847 .20786 Mdot
.44206 .81144 Mdot
.5662 .22206 Mdot
.58953 .19513 Mdot
.48846 .85276 Mdot
.47344 .84108 Mdot
.52299 .27258 Mdot
.41174 .7731 Mdot
.39837 .75122 Mdot
.42649 .7933 Mdot
.55279 .23771 Mdot
.5383 .25462 Mdot
.53716 .88458 Mdot
.6294 .15066 Mdot
.62271 .1573 Mdot
.63589 .14491 Mdot
.55393 .89544 Mdot
.54615 .89023 Mdot
.50256 .86264 Mdot
.60768 .17366 Mdot
.59917 .18374 Mdot
.61544 .16488 Mdot
.52703 .87825 Mdot
.51549 .87104 Mdot
.42036 .43299 Mdot
.35631 .58058 Mdot
.35613 .56457 Mdot
.35698 .59794 Mdot
.43367 .40226 Mdot
.42631 .41811 Mdot
.40953 .47288 Mdot
.35637 .53566 Mdot
.35656 .52234 Mdot
.3562 .54967 Mdot
.41566 .44698 Mdot
.41206 .46024 Mdot
.47803 .33002 Mdot
.37735 .70466 Mdot
.37 .68128 Mdot
.38682 .72819 Mdot
.50753 .29138 Mdot
.49242 .31066 Mdot
.44251 .38543 Mdot
.36083 .63701 Mdot
.35839 .61675 Mdot
.3646 .65861 Mdot
.46468 .34912 Mdot
.45283 .36767 Mdot
.03298 .60191 Mdot
.97538 .52511 Mdot
.96206 .39601 Mdot
.94794 .64614 Mdot
.21062 .09909 Mdot
.0663 .28622 Mdot
.39879 .97438 Mdot
.87376 .23233 Mdot
.83378 .19273 Mdot
.91985 .29637 Mdot
.15295 .84922 Mdot
.29667 .94908 Mdot
.51696 .03527 Mdot
.76354 .85306 Mdot
.78922 .83952 Mdot
.74166 .86192 Mdot
.55457 .05275 Mdot
.54 .04418 Mdot
.34046 .03631 Mdot
.85786 .78482 Mdot
.90207 .73143 Mdot
.82045 .81812 Mdot
.48204 .02733 Mdot
.42703 .02389 Mdot
.57162 .91583 Mdot
.67535 .12876 Mdot
.66575 .13034 Mdot
.6863 .12816 Mdot
.57115 .92783 Mdot
.57248 .92154 Mdot
.5604 .90043 Mdot
.64986 .13615 Mdot
.64262 .14009 Mdot
.65743 .13287 Mdot
.56911 .91053 Mdot
.56538 .90543 Mdot
.53239 .9592 Mdot
.77276 .15142 Mdot
.74942 .14091 Mdot
.80035 .1675 Mdot
.46409 .97485 Mdot
.5056 .96794 Mdot
.5676 .93467 Mdot
.71299 .13048 Mdot
.69859 .12855 Mdot
.72995 .13448 Mdot
.54974 .95044 Mdot
.56104 .94217 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[34]:=",
  ImageSize->{288, 288},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181000`40n007h00Ooooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00
oooo8Ooo000Zool20004ool00`00ooooo`02ool4000Sool20004ool00`00ooooo`02ool4000Vool2
000Zool20004ool00`00ooooo`02ool4000Sool20004ool00`00ooooo`02ool4000Rool002Woo`04
003ooooo0008ool00`00ooooo`0Sool01000ooooo`002_oo00<00?ooool09?oo00@00?ooool002So
o`04003ooooo000:ool00`00ooooo`0Qool01000ooooo`002?oo00<00?ooool08ooo000Yool01000
ooooo`002Ooo00<00?ooool08_oo00@00?ooool000[oo`03003ooooo02Coo`04003ooooo000Xool0
1000ooooo`002_oo00<00?ooool08Ooo00@00?ooool000Woo`03003ooooo02;oo`008ooo10000_oo
00@00?ooool000[oo`03003ooooo01_oo`@000;oo`04003ooooo000:ool00`00ooooo`0Tool01000
ooooo`00:?oo00@00?ooool000[oo`03003ooooo027oo`04003ooooo000:ool00`00ooooo`0Qool0
02Woo`04003ooooo000;ool00`00ooooo`0Pool01000ooooo`002_oo00<00?ooool09?oo00@00?oo
ool002Soo`04003ooooo000:ool00`00ooooo`0Qool01000ooooo`002ooo00<00?ooool08?oo000Y
ool01000ooooo`002?oo00@00?ooool002;oo`04003ooooo0008ool3000Vool01000ooooo`00:?oo
00@00?ooool000Soo`<002?oo`04003ooooo0008ool01000ooooo`008_oo000Zool2000:ool2000T
ool2000;ool00`00ooooo`0Uool2000Zool2000;ool00`00ooooo`0Rool2000:ool2000Sool00?oo
ob7oo`00oooo8Ooo003ooolQool001_oool000D0007oo`006ooo00<00?ooool01_oo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool0
1ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo0@000Ooo000K
ool00`00ooooo`0@ool00`00ooooo`0_ool00`00ooooo`0_ool00`00ooooo`0_ool00`00ooooo`0_
ool00`00ooooo`0Uool10001ool001_oo`03003ooooo0?ooo`7oo`40007oo`006ooo00<00?ooool0
oooo0Ooo0@000Ooo000Kool00`00ooooo`3oool1ool10001ool001_oo`8006Gooa@008Ooo`80007o
o`006ooo00<00?ooool0G_oo8000P_oo0@000Ooo000Kool00`00ooooo`1Hool600001?oo00000?oo
0`004ooo2`00OOoo0@000Ooo000Kool00`00ooooo`1Eool300001?oo00000?oo0P008?oo2000N_oo
0@000Ooo000Kool00`00ooooo`1Aool400000ooo0000000Zool7001gool10001ool001_oo`03003o
oooo04koo`<00004ool00?oo000`ool6001eool10001ool001_oo`03003ooooo04_oo`<00003ool0
000003Ooo`D007?oo`40007oo`006ooo00<00?ooool0B?oo1P00??oo1000L_oo0@000Ooo000Kool0
0`00ooooo`16ool400000ooo003oo`0oool3001aool10001ool001_oo`03003ooooo04Coo`800004
ool00?oo0013ool4001_ool10001ool001_oo`8004?oo`800003ool00?oo04Soo`8006koo`80007o
o`006ooo00<00?ooool0@?oo0P0000?oo`00ool0B_oo0`00K_oo0@000Ooo000Kool00`00ooooo`11
ool00`00ooooo`1<ool2001^ool10001ool001_oo`03003ooooo03koo`80057oo`8006goo`40007o
o`006ooo00<00?ooool0??oo0P0000?oo`00ool0D?oo0P00KOoo0@000Ooo000Kool00`00ooooo`0i
ool20002ool00`00ooooo`1Aool2001]ool10001ool001_oo`03003ooooo03Ooo`800003ool00?oo
05Coo`<006goo`40007oo`006ooo00<00?ooool0>?oo00<00?ooool0EOoo0`00KOoo0@000Ooo000K
ool00`00ooooo`0eool2001Hool3001^ool10001ool000Soo`8000Coo`03003ooooo00;oo`@000Co
o`03003ooooo03Koo`03003ooooo05Goo`@006koo`40007oo`001ooo00@00?ooool000Soo`03003o
oooo00Goo`03003ooooo03?oo`8005Soo`@006ooo`40007oo`001ooo00@00?ooool000Woo`03003o
oooo00Coo`8003Goo`03003ooooo05Coo`D006ooo`80007oo`0000?oo`0000000P000_oo00@00?oo
ool000[oo`03003ooooo00?oo`03003ooooo033oo`8005Soo`D0077oo`40007oo`001ooo00@00?oo
ool000_oo`03003ooooo00;oo`03003ooooo02koo`800003ool00?oo05Goo`D007?oo`40007oo`00
1ooo00@00?ooool000Soo`04003ooooo0004ool00`00ooooo`0_ool00`00ooooo`1Eool5001dool1
0001ool000Soo`8000[oo`8000Goo`03003ooooo08Goo`D007Koo`40007oo`006ooo00<00?ooool0
:ooo0P00E_oo1@00N?oo0@000Ooo000Kool00`00ooooo`0Zool3001Eool5000Sool<001:ool10001
ool001_oo`03003ooooo02_oo`03003ooooo05;oo`D0027ooaD004Goo`40007oo`006ooo00<00?oo
ool0:?oo0P00EOoo1@008?oo20002Ooo2@00@_oo0@000Ooo000Kool00`00ooooo`0Wool3001Cool5
000Qool5000Bool7000oool10001ool001_oo`8002Woo`03003ooooo057oo`D0023oo`D001Soo`D0
03coo`80007oo`006ooo00<00?ooool09Ooo0P00E?oo10008Ooo1@007?oo1000>ooo0@000Ooo000K
ool00`00ooooo`0Vool00`00ooooo`1@ool5000Qool4000Pool4000iool10001ool001_oo`03003o
oooo02?oo`8005?oo`@002;oo`@002?oo`@003Ooo`40007oo`006ooo00<00?ooool08_oo0`00D_oo
10008_oo10009_oo1000=Ooo0@000Ooo000Kool00`00ooooo`0Sool00`00ooooo`1@ool4000Rool4
000Yool4000cool10001ool001_oo`03003ooooo023oo`8005?oo`@002;oo`@002_oo`@003;oo`40
007oo`006ooo00<00?ooool08Ooo00<00?ooool0D?oo0`009?oo0`00;_oo0`00<Ooo0@000Ooo000K
ool00`00ooooo`1bool4000Tool3000`ool4000_ool10001ool001_oo`03003ooooo01koo`80057o
o`@002Coo`<003?oo`<002koo`40007oo`006ooo0P008?oo00<00?ooool0C_oo10009?oo0`00=Ooo
1000:ooo0P000Ooo000Kool00`00ooooo`1_ool4000Tool4000gool3000[ool10001ool001_oo`03
003ooooo01_oo`80057oo`@002Coo`@003Woo`<002[oo`40007oo`006ooo00<00?ooool07?oo00<0
0?ooool0C_oo10009?oo1000??oo0P00:Ooo0@000Ooo000Kool00`00ooooo`1]ool3000Tool4000n
ool2000Xool10001ool001_oo`03003ooooo01[oo`80053oo`<002Goo`<0043oo`8002Ooo`40007o
o`006ooo00<00?ooool06ooo00<00?ooool0COoo0`009Ooo0`00@_oo0`009Ooo0@000Ooo000Kool0
0`00ooooo`1Zool4000Tool30014ool2000Uool10001ool001_oo`03003ooooo06Woo`@002Coo`@0
04Goo`<002?oo`40007oo`006ooo00<00?ooool05ooo0P00D?oo0`009?oo1000Aooo0`008_oo0@00
0Ooo000Kool00`00ooooo`0Hool00`00ooooo`1=ool3000Tool40019ool2000Rool10001ool001_o
o`8006Woo`<002?oo`@004_oo`80023oo`80007oo`006ooo00<00?ooool05Ooo0P00D?oo0`009?oo
0`00COoo0P008?oo0@000Ooo000Kool00`00ooooo`0Fool00`00ooooo`1=ool3000Tool3001?ool2
000Oool10001ool001_oo`03003ooooo06Goo`<002Coo`<0057oo`8001koo`40007oo`006ooo00<0
0?ooool0IOoo0`008ooo1000DOoo0`007Ooo0@000Ooo000Kool00`00ooooo`0Cool2001?ool3000S
ool4001Cool3000Lool10001ool001_oo`03003ooooo01Coo`03003ooooo04goo`<002;oo`@005Go
o`8001coo`40007oo`006ooo00<00?ooool0Hooo0`008_oo1000Eooo0P006ooo0@000Ooo000Kool0
0`00ooooo`0Aool2001@ool3000Rool3001Iool2000Jool10001ool001_oo`03003ooooo01;oo`03
003ooooo04goo`<002;oo`<005_oo`8001Woo`40007oo`006ooo0P00Hooo0`008Ooo0`00G?oo0P00
6?oo0P000Ooo000Kool00`00ooooo`1Qool3000Qool4001Mool2000Hool10001ool001_oo`03003o
oooo067oo`80027oo`@005ooo`8001Ooo`40007oo`006ooo00<00?ooool03ooo0P00Cooo0`008Ooo
0`00H?oo0`005_oo0@000Ooo000Kool00`00ooooo`0@ool00`00ooooo`1=ool3000Pool3001Rool2
000Fool10001ool001_oo`03003ooooo05ooo`<0023oo`<006Coo`8001Goo`40007oo`006ooo00<0
0?ooool03Ooo0P00D?oo0`007ooo0`00I_oo0P005?oo0@000Ooo000Kool00`00ooooo`0>ool00`00
ooooo`1>ool2000Oool3001Wool2000Dool10001ool000Soo`8000Coo`03003ooooo00;oo`@000Co
o`03003ooooo05koo`<001koo`@006Soo`03003ooooo01;oo`40007oo`001ooo00@00?ooool000[o
o`03003ooooo00?oo`03003ooooo05koo`<001goo`@006Woo`8001?oo`40007oo`001ooo00@00?oo
ool000[oo`03003ooooo00?oo`8005ooo`8001koo`<006_oo`80017oo`80007oo`0000?oo`000000
0P000_oo00@00?ooool000[oo`03003ooooo00?oo`03003ooooo00_oo`80053oo`<001goo`<006go
o`80017oo`40007oo`001ooo00@00?ooool000[oo`03003ooooo00?oo`03003ooooo00coo`03003o
oooo04koo`8001goo`<006koo`80017oo`40007oo`001ooo00@00?ooool000Soo`<000Goo`03003o
oooo05goo`8001coo`@006ooo`03003ooooo00ooo`40007oo`002?oo0P002ooo00<00?ooool00ooo
00<00?ooool0G?oo0`007?oo0`00L?oo0P004?oo0@000Ooo000Kool00`00ooooo`0:ool2001@ool3
000Kool3001bool2000?ool10001ool001_oo`03003ooooo00_oo`03003ooooo04koo`8001_oo`<0
07Coo`03003ooooo00goo`40007oo`006ooo00<00?ooool0G?oo0P006_oo1000Looo0P003ooo0@00
0Ooo000Kool00`00ooooo`1Kool3000Jool3001eool2000>ool10001ool001_oo`03003ooooo05_o
o`<001Woo`<007Ooo`03003ooooo00coo`40007oo`006ooo00<00?ooool02?oo0P00DOoo0P006Ooo
1000Mooo0P003Ooo0@000Ooo000Kool2000:ool00`00ooooo`1?ool2000Iool3001hool2000<ool2
0001ool001_oo`03003ooooo05_oo`8001Soo`<007[oo`8000coo`40007oo`006ooo00<00?ooool0
F_oo0`005ooo0`00Nooo0P003?oo0@000Ooo000Kool00`00ooooo`07ool2001Aool2000Hool3001l
ool2000;ool10001ool001_oo`03003ooooo00Soo`03003ooooo04ooo`8001Ooo`<007koo`03003o
oooo00Woo`40007oo`006ooo00<00?ooool0F_oo0P005_oo1000OOoo0P002ooo0@000Ooo000Kool0
0`00ooooo`1Jool2000Fool3001oool00`00ooooo`09ool10001ool001_oo`03003ooooo00Koo`80
05;oo`8001Goo`<0083oo`8000[oo`40007oo`006ooo00<00?ooool01ooo00<00?ooool0Cooo0`00
5?oo1000P?oo0P002_oo0@000Ooo000Kool00`00ooooo`06ool2001Aool3000Dool30022ool00`00
ooooo`08ool10001ool001_oo`8000Soo`03003ooooo04ooo`8001Coo`<008?oo`8000Soo`80007o
o`006ooo00<00?ooool0FOoo0P005?oo0`00Pooo0P002Ooo0@000Ooo000Kool00`00ooooo`1Iool2
000Cool30025ool00`00ooooo`07ool10001ool001_oo`03003ooooo00Goo`8005;oo`8001?oo`<0
08ooo`40007oo`006ooo00<00?ooool01_oo00<00?ooool0D?oo0P004_oo0`00Q_oo0P002?oo0@00
0Ooo000Kool00`00ooooo`1Iool2000Bool30026ool20008ool10001ool001_oo`03003ooooo00Co
o`8005?oo`80017oo`<008Soo`03003ooooo00Koo`40007oo`006ooo00<00?ooool01Ooo00<00?oo
ool0DOoo0P004Ooo0`00R?oo0P001ooo0@000Ooo000Kool00`00ooooo`04ool2001Cool2000@ool3
002:ool00`00ooooo`05ool10001ool001_oo`03003ooooo00Goo`03003ooooo057oo`80013oo`<0
08Woo`8000Ooo`40007oo`006ooo0P00F_oo0P004?oo0P00Rooo00<00?ooool01?oo0P000Ooo000K
ool00`00ooooo`1Iool2000?ool3002;ool20006ool10001ool001_oo`03003ooooo00Coo`8005?o
o`8000ooo`8008goo`03003ooooo00Coo`40007oo`006ooo00<00?ooool01Ooo00<00?ooool0DOoo
0P003_oo0`00S?oo0P001_oo0@000Ooo000Kool00`00ooooo`1Iool2000>ool3002=ool00`00oooo
o`04ool10001ool001_oo`03003ooooo05Woo`8000koo`8008goo`8000Koo`40007oo`006ooo00<0
0?ooool00ooo0P00E?oo0P003Ooo0`00S_oo00<00?ooool01?oo0@000Ooo000Kool00`00ooooo`04
ool00`00ooooo`1Bool2000=ool3002>ool20005ool10001ool001_oo`03003ooooo05Woo`8000go
o`80093oo`03003ooooo00?oo`40007oo`006ooo00<00?ooool0FOoo0P003?oo0`00Sooo0P001Ooo
0@000Ooo000Kool00`00ooooo`1Iool2000<ool3002@ool00`00ooooo`03ool10001ool001_oo`80
05[oo`8000coo`80093oo`8000Coo`80007oo`006ooo00<00?ooool0FOoo0P003?oo0P00T?oo0P00
1Ooo0@000Ooo000Kool00`00ooooo`03ool2001Dool2000<ool2002@ool20005ool10001ool001_o
o`03003ooooo00Coo`03003ooooo05;oo`8000_oo`<0097oo`03003ooooo00?oo`40007oo`006ooo
00<00?ooool0FOoo0P002ooo0`00TOoo0P001?oo0@000Ooo000Kool00`00ooooo`1Iool2000;ool2
002Bool20004ool10001ool001_oo`03003ooooo00;oo`8005Goo`8000_oo`8009;oo`8000Coo`40
007oo`006ooo00<00?ooool00ooo00<00?ooool0Dooo0P002ooo0P00Tooo00<00?ooool00_oo0@00
0Ooo000Dool20005ool00`00ooooo`1Iool2000;ool2002Bool20004ool10001ool001?oo`04003o
oooo0004ool00`00ooooo`1Iool2000;ool2002Cool00`00ooooo`02ool10001ool001?oo`04003o
oooo0004ool2001Jool2000;ool2002Bool20003ool20001ool001?oo`04003ooooo0004ool00`00
ooooo`1Iool2000;ool2002Cool00`00ooooo`02ool10001ool001?oo`04003ooooo0004ool00`00
ooooo`02ool2001Eool2000;ool2002Bool20004ool10001ool001?oo`04003ooooo0004ool00`00
ooooo`03ool00`00ooooo`1Cool2000;ool2002Cool00`00ooooo`02ool10001ool001Coo`8000Go
o`03003ooooo05Woo`8000_oo`8009;oo`8000Coo`40007oo`006ooo00<00?ooool0FOoo0P002ooo
0P00Tooo00<00?ooool00_oo0@000Ooo000Kool00`00ooooo`03ool2001Dool2000;ool3002@ool2
0005ool10001ool001_oo`03003ooooo00Coo`03003ooooo05;oo`8000coo`80097oo`03003ooooo
00?oo`40007oo`006ooo00<00?ooool0FOoo0P003?oo0P00T?oo0P001Ooo0@000Ooo000Kool00`00
ooooo`1Iool2000<ool2002Aool00`00ooooo`03ool10001ool001_oo`8005[oo`8000coo`80093o
o`8000Coo`80007oo`006ooo00<00?ooool0FOoo0P003?oo0`00T?oo00<00?ooool00ooo0@000Ooo
000Kool00`00ooooo`03ool2001Dool2000=ool2002?ool20005ool10001ool001_oo`03003ooooo
00Coo`03003ooooo05;oo`8000goo`80093oo`03003ooooo00?oo`40007oo`006ooo00<00?ooool0
FOoo0P003Ooo0P00Sooo0P001Ooo0@000Ooo000Kool00`00ooooo`1Iool2000>ool2002?ool00`00
ooooo`03ool10001ool001_oo`03003ooooo00?oo`8005Coo`8000koo`8008goo`8000Koo`40007o
o`006ooo00<00?ooool01?oo00<00?ooool0D_oo0P003_oo0`00SOoo00<00?ooool01?oo0@000Ooo
000Kool00`00ooooo`1Iool2000?ool2002<ool20006ool10001ool001_oo`03003ooooo05Woo`80
00ooo`8008goo`03003ooooo00Coo`40007oo`006ooo00<00?ooool0FOoo0P003ooo0`00Rooo0P00
1_oo0@000Ooo000Kool2001Jool2000@ool2002<ool00`00ooooo`03ool20001ool001_oo`03003o
oooo00Coo`8005?oo`80013oo`<008Woo`8000Ooo`40007oo`006ooo00<00?ooool01Ooo00<00?oo
ool0DOoo0P004Ooo0P00R_oo00<00?ooool01Ooo0@000Ooo000Kool00`00ooooo`1Iool2000Aool3
0028ool20007ool10001ool001_oo`03003ooooo05Woo`8001;oo`8008Ooo`<000Ooo`40007oo`00
6ooo00<00?ooool01Ooo0P00D_oo0P004_oo0`00Q_oo0P002?oo0@000Ooo000Kool00`00ooooo`06
ool00`00ooooo`1@ool2000Cool20026ool20008ool10001ool001_oo`03003ooooo05Woo`8001?o
o`<008Koo`03003ooooo00Koo`40007oo`006ooo00<00?ooool0FOoo0P005?oo0P00Q?oo0P002Ooo
0@000Ooo000Kool00`00ooooo`1Iool2000Dool30024ool00`00ooooo`07ool10001ool001_oo`80
05[oo`8001Goo`8008?oo`8000Soo`80007oo`006ooo00<00?ooool01_oo0P00DOoo0`005?oo0`00
POoo0`002Ooo0@000Ooo000Kool00`00ooooo`07ool00`00ooooo`1@ool2000Eool20022ool00`00
ooooo`08ool10001ool001_oo`03003ooooo05[oo`8001Koo`80083oo`8000[oo`40007oo`006ooo
00<00?ooool0F_oo0P005_oo0`00P?oo00<00?ooool02?oo0@000Ooo000Kool00`00ooooo`07ool2
001Aool2000Gool2001nool2000;ool10001ool001_oo`03003ooooo00Soo`03003ooooo04ooo`80
01Soo`8007koo`03003ooooo00Woo`40007oo`006ooo00<00?ooool0F_oo0P006?oo0`00Nooo0P00
3?oo0@000Ooo000Kool00`00ooooo`1Jool3000Hool2001kool2000<ool10001ool001_oo`03003o
oooo05_oo`8001Soo`<007_oo`03003ooooo00[oo`40007oo`006ooo0P002Ooo0P00DOoo0P006Ooo
0`00N?oo0P003?oo0P000Ooo000Kool00`00ooooo`09ool00`00ooooo`1?ool2000Jool2001hool2
000=ool10001ool001_oo`03003ooooo05_oo`8001_oo`8007Soo`03003ooooo00_oo`40007oo`00
6ooo00<00?ooool0G?oo0P006_oo0`00MOoo0P003_oo0@000Ooo000Kool00`00ooooo`0:ool2001@
ool2000Kool3001cool3000>ool10001ool001_oo`03003ooooo00_oo`03003ooooo04koo`8001co
o`8007Coo`03003ooooo00goo`40007oo`006ooo00<00?ooool0G?oo0`006ooo0`00LOoo0P004?oo
0@000Ooo000Kool00`00ooooo`0;ool2001@ool2000Lool3001`ool2000@ool10001ool001_oo`03
003ooooo00coo`03003ooooo04koo`8001goo`8006ooo`<0013oo`40007oo`002?oo0P001?oo00<0
0?ooool00_oo10001?oo00<00?ooool03?oo0P00Cooo0P007_oo0P00KOoo0`004Ooo0@000Ooo0007
ool01000ooooo`002_oo00<00?ooool00ooo00<00?ooool03Ooo00<00?ooool0COoo0`007_oo0P00
K?oo0P004_oo0@000Ooo0007ool01000ooooo`002_oo00<00?ooool00ooo0P00Gooo0P007_oo0`00
J_oo0`004Ooo0P000Ooo0007ool01000ooooo`002_oo00<00?ooool00ooo00<00?ooool0G_oo0`00
7_oo0`00JOoo0P004ooo0@000Ooo0007ool01000ooooo`002_oo00<00?ooool00ooo00<00?ooool0
3Ooo0P00D?oo0P007ooo0P00J?oo0`004ooo0@000Ooo0007ool01000ooooo`002?oo0`001Ooo00<0
0?ooool03_oo00<00?ooool0C_oo0P008?oo0P00I_oo0`005?oo0@000Ooo0008ool2000;ool00`00
ooooo`03ool00`00ooooo`1Oool3000Pool2001Uool2000Eool10001ool001_oo`03003ooooo00ko
o`80053oo`80023oo`<006Goo`03003ooooo01?oo`40007oo`006ooo00<00?ooool03ooo00<00?oo
ool0C_oo0P008Ooo0`00HOoo0`005_oo0@000Ooo000Kool00`00ooooo`0?ool2001@ool2000Qool3
001Qool2000Fool10001ool001_oo`03003ooooo013oo`03003ooooo04koo`8002;oo`<005koo`80
01Soo`40007oo`006ooo00<00?ooool0HOoo0`008_oo0P00G_oo0P006?oo0@000Ooo000Kool2000B
ool2001?ool2000Sool2001Kool4000Gool20001ool001_oo`03003ooooo01;oo`03003ooooo04ko
o`8002;oo`<005_oo`8001Woo`40007oo`006ooo00<00?ooool0Hooo0P008ooo0`00F?oo0`006_oo
0@000Ooo000Kool00`00ooooo`0Bool2001@ool2000Sool3001Fool4000Jool10001ool001_oo`03
003ooooo01?oo`03003ooooo04koo`8002Coo`<005Goo`8001coo`40007oo`006ooo00<00?ooool0
IOoo0P009?oo0`00Dooo0`007?oo0@000Ooo000Kool00`00ooooo`1Uool3000Tool2001Bool3000M
ool10001ool001_oo`03003ooooo01Goo`8004ooo`8002Goo`80053oo`<001koo`40007oo`006ooo
00<00?ooool05_oo00<00?ooool0C_oo0P009?oo0`00C_oo0`007ooo0@000Ooo000Kool00`00oooo
o`1Wool3000Tool3001<ool3000Pool10001ool001_oo`8001Ooo`80053oo`8002Goo`<004[oo`@0
01ooo`80007oo`006ooo00<00?ooool05ooo00<00?ooool0Cooo0P009Ooo0`00BOoo0P008_oo0@00
0Ooo000Kool00`00ooooo`1Yool3000Uool20017ool4000Rool10001ool001_oo`03003ooooo06[o
o`<002Goo`8004Goo`@002?oo`40007oo`006ooo00<00?ooool06Ooo0P00D?oo0P009_oo0P00@ooo
10009?oo0@000Ooo000Kool00`00ooooo`0Jool00`00ooooo`1?ool2000Uool30011ool3000Vool1
0001ool001_oo`03003ooooo06coo`<002Goo`<003ooo`<002Ooo`40007oo`006ooo00<00?ooool0
6ooo0P00D?oo0`009Ooo0`00?Ooo10009ooo0@000Ooo000Kool00`00ooooo`0Lool00`00ooooo`1?
ool3000Uool3000kool4000Xool10001ool001_oo`03003ooooo06ooo`<002Goo`<003Soo`D002Wo
o`40007oo`006ooo00<00?ooool0L?oo0`009Ooo0P00=ooo1000:ooo0@000Ooo000Kool2000Nool2
001Bool3000Uool2000eool4000[ool20001ool001_oo`03003ooooo01koo`03003ooooo057oo`<0
02Coo`<003?oo`@002goo`40007oo`006ooo00<00?ooool07ooo0P00D_oo0`009?oo0`00<?oo1000
;ooo0@000Ooo000Kool00`00ooooo`0Pool00`00ooooo`1Aool3000Tool3000]ool5000`ool10001
ool001_oo`03003ooooo07Goo`<002Coo`<002_oo`@003;oo`40007oo`006ooo00<00?ooool08_oo
0P00D_oo0`009?oo0`00:?oo1@00<ooo0@000Ooo000Kool00`00ooooo`0Sool00`00ooooo`1Aool4
000Sool3000Uool5000eool10001ool001_oo`03003ooooo02Coo`8005?oo`<002?oo`<002;oo`D0
03Ooo`40007oo`006ooo00<00?ooool09Ooo00<00?ooool0D_oo0`008ooo0`007ooo1P00>?oo0@00
0Ooo000Kool00`00ooooo`1kool4000Rool3000Lool6000jool10001ool001_oo`8002Soo`8005Co
o`<002;oo`@001Ooo`L003_oo`80007oo`006ooo00<00?ooool0:?oo00<00?ooool0Dooo10008Ooo
1@004_oo1`00?ooo0@000Ooo000Kool00`00ooooo`0Yool2001Dool4000Rool6000;ool90011ool1
0001ool001_oo`03003ooooo02[oo`03003ooooo05Coo`@002;ooaD004Coo`40007oo`006ooo00<0
0?ooool0P_oo10009?oo3@00BOoo0@000Ooo000Kool00`00ooooo`0/ool2001Fool4001hool10001
ool001_oo`03003ooooo02goo`8005Ooo`@007Koo`40007oo`006ooo00<00?ooool0;_oo00<00?oo
ool0E_oo1000MOoo0@000Ooo0008ool20004ool00`00ooooo`02ool40004ool00`00ooooo`0_ool2
001Hool4001cool10001ool000Ooo`04003ooooo0008ool00`00ooooo`05ool00`00ooooo`0`ool2
001Hool4001bool10001ool000Ooo`04003ooooo0009ool00`00ooooo`04ool2000bool3001Hool4
001_ool20001ool000Ooo`04003ooooo000:ool00`00ooooo`03ool00`00ooooo`0cool3001Gool4
001_ool10001ool000Ooo`04003ooooo000;ool00`00ooooo`02ool00`00ooooo`0eool2001Hool3
001^ool10001ool000Ooo`04003ooooo0008ool01000ooooo`001?oo00<00?ooool0=_oo0`00Eooo
0P00K_oo0@000Ooo0008ool2000:ool20005ool00`00ooooo`0hool00`00ooooo`1Eool3001]ool1
0001ool001_oo`03003ooooo03Soo`8005Ooo`8006goo`40007oo`006ooo00<00?ooool0>Ooo00@0
0?oo000005Coo`8006goo`40007oo`006ooo00<00?ooool0??oo0`00D_oo0P00KOoo0@000Ooo000K
ool00`00ooooo`0nool00`00ooooo`1?ool3001]ool10001ool001_oo`03003ooooo043oo`8004ko
o`<006goo`40007oo`006ooo00<00?ooool0@Ooo0P00C?oo0`00K_oo0@000Ooo000Kool20013ool0
1000ool00000B?oo1000KOoo0P000Ooo000Kool00`00ooooo`15ool20016ool4001_ool10001ool0
01_oo`03003ooooo04Koo`03003oo`000080043oo`D0073oo`40007oo`006ooo00<00?ooool0BOoo
0P0000?oo`000000>ooo1P00LOoo0@000Ooo000Kool00`00ooooo`1=ool4000gool5001cool10001
ool001_oo`03003ooooo04koo`03003oo`0000@0033oo`H007Goo`40007oo`006ooo00<00?ooool0
D_oo1P00:_oo1`00Mooo0@000Ooo000Kool00`00ooooo`1Eool7000Rool8001jool10001ool001_o
o`03003ooooo05Soo`03003oo`0000P001Goo`/007goo`40007oo`006ooo00<00?ooool0GOoo0P00
00?oo`0000007@00POoo0@000Ooo000Kool2001Uool400000ooo0000000?0025ool20001ool001_o
o`03003ooooo0?ooo`7oo`40007oo`006ooo00<00?ooool0oooo0Ooo0@000Ooo000Kool00`00oooo
o`3oool1ool10001ool001_oo`03003ooooo013oo`03003ooooo02ooo`03003ooooo02ooo`03003o
oooo02ooo`03003ooooo02ooo`03003ooooo02Goo`40007oo`006ooo00<00?ooool01_oo00<00?oo
ool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?oo
ool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?oo
ool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?oo
ool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?oo
ool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo00<00?ooool01ooo0@000Ooo
000Koooo00050001ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQ
ool00?ooob7oo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-0.293698, -0.289638, \
0.00199935, 0.00193578}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["The ABC flow", "Subsubsection"],

Cell["\<\
This loads an example problem of the \
Arnold\[Dash]Beltrami\[Dash]Childress (ABC) flow which is used to model chaos \
in  laminar flows of the three-dimensional Euler equations.\
\>", "Text"],

Cell[BoxData[{
    \(\(system = 
        GetNDSolveProblem["\<ArnoldBeltramiChildress\>"];\)\), "\
\[IndentingNewLine]", 
    \(\(eqs\  = \ system["\<System\>"[]];\)\), "\[IndentingNewLine]", 
    \(\(vars\  = \ 
        system["\<DependentVariables\>"[]];\)\), "\[IndentingNewLine]", 
    \(\(icvars\  = \ vars\  /. \ T \[Rule] 0;\)\)}], "Input",
  CellLabel->"In[25]:="],

Cell["\<\
This defines a splitting Y1, Y2 of the system by setting some of \
the right-hand side components to zero.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Y1 = eqs; \ Y1[\([2, 2]\)]\  = \ 0; Y1\)], "Input",
  CellLabel->"In[29]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_1\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], 
          "\[Equal]", \(3\/4\ Cos[Y\_2[T]] + Sin[Y\_3[T]]\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_2\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_3\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], 
          "\[Equal]", \(Cos[Y\_1[T]] + 3\/4\ Sin[Y\_2[T]]\)}]}], 
      "}"}]], "Output",
  CellLabel->"Out[29]="]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Y2 = eqs; Y2[\([{1, 3}, 2]\)]\  = \ 0; \ Y2\)], "Input",
  CellLabel->"In[30]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_1\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_2\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], 
          "\[Equal]", \(Cos[Y\_3[T]] + Sin[Y\_1[T]]\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_3\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}]}], 
      "}"}]], "Output",
  CellLabel->"Out[30]="]
}, Open  ]],

Cell["This defines the implicit midpoint method.", "MathCaption"],

Cell[BoxData[
    \(\(ImplicitMidpoint\  = \ {"\<ImplicitRungeKutta\>", \ \
"\<Coefficients\>" \[Rule] "\<ImplicitRungeKuttaGaussCoefficients\>", \
"\<DifferenceOrder\>" \[Rule] 
            2, "\<ImplicitSolver\>" \[Rule] {"\<FixedPoint\>", 
              AccuracyGoal \[Rule] 10, \ 
              PrecisionGoal \[Rule] 10, "\<IterationSafetyFactor\>" \[Rule] 
                1}};\)\)], "Input",
  CellLabel->"In[31]:="],

Cell["\<\
This constructs a second-order splitting method that retains volume \
and reversing symmetries.\
\>", "MathCaption"],

Cell[BoxData[{
    \(\(\(StrangSplittingCoefficients[2, \ p_] := \ 
        N[{1/2, 1, 1/2}, \ p];\)\(\[IndentingNewLine]\)
    \)\), "\[IndentingNewLine]", 
    \(\(ABCSplitting\  = \ {"\<Splitting\>", \[IndentingNewLine]\ \
"\<Coefficients\>" \[Rule] 
            StrangSplittingCoefficients, \
\[IndentingNewLine]"\<DifferenceOrder\>" \[Rule] 
            2, \[IndentingNewLine]"\<Equations\>" \[Rule] {Y2, Y1, 
              Y2}, \[IndentingNewLine]"\<Method\>" \[Rule] \
{"\<LocallyExact\>", ImplicitMidpoint, "\<LocallyExact\>"}};\)\)}], "Input",
  CellLabel->"In[32]:="],

Cell["\<\
This defines a function which gives the Poincar\[EAcute] section \
for a particular initial condition.\
\>", "MathCaption"],

Cell[BoxData[
    \(\(psect[
          ics_]\  := \ \[IndentingNewLine]Module[{reapdata}, \
\[IndentingNewLine]reapdata\  = \ \
\[IndentingNewLine]Reap[\[IndentingNewLine]NDSolve[{eqs, 
                  Thread[icvars\  \[Equal] \ ics]}, {}, {T, 
                  1000}, \[IndentingNewLine]\ 
                Method \[Rule] {EventLocator, \[IndentingNewLine]"\<Event\>" \
\[Rule] Y\_2[T], \ "\<EventAction\>" \[RuleDelayed] \ 
                      Sow[{Y\_1[T], \ 
                          Y\_3[T]}], \ \
\[IndentingNewLine]"\<EventLocationMethod\>" -> "\<LinearInterpolation\>", 
                    Method \[Rule] 
                      ABCSplitting}, \ \[IndentingNewLine]StartingStepSize \
\[Rule] 1/4, 
                MaxSteps \[Rule] \[Infinity]]\[IndentingNewLine]]; \
\[IndentingNewLine]reapdata[\([\(-1\), 
              1]\)]\[IndentingNewLine]];\)\)], "Input",
  CellLabel->"In[34]:="],

Cell["\<\
This finds the Poincar\[EAcute] sections for several different \
initial conditions and flattens them together into a single list of points.\
\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(data\  = \ \[IndentingNewLine]Flatten[\[IndentingNewLine]Mod[
            Map[psect, \ {{4.267682454609692, 0, 
                  0.9952906114885919}, \
\[IndentingNewLine]{1.6790790859443243, 0, 
                  2.1257099470901704}, \
\[IndentingNewLine]{2.9189523719753327, 0, 
                  4.939152797323216}, \[IndentingNewLine]{3.1528896559036776, 
                  0, 4.926744120488727}, \
\[IndentingNewLine]{0.9829282640373566, 0, 
                  1.7074633238173198}, \
\[IndentingNewLine]{0.4090394012299985, 0, 
                  4.170087631574883}, \[IndentingNewLine]{6.090600411133905, 
                  0, 2.3736566160602277}, \
\[IndentingNewLine]{6.261716134007686, 0, 
                  1.4987884558838156}, \[IndentingNewLine]{1.005126683795467, 
                  0, 1.3745418575363608}, \
\[IndentingNewLine]{1.5880780704325377, 0, 
                  1.3039536044289253}, \[IndentingNewLine]{3.622408133554125, 
                  0, 2.289597511313432}, \
\[IndentingNewLine]{0.030948690635763183, 0, 
                  4.306922133429981}, \[IndentingNewLine]{5.906038850342371, 
                  0, 5.000045498132029}}], \[IndentingNewLine]2  \[Pi]], \ \
\[IndentingNewLine]1];\)\), "\[IndentingNewLine]", 
    \(\(ListPlot[data];\)\)}], "Input",
  CellLabel->"In[35]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.15168 0.0147151 0.103151 [
[.17549 .00222 -3 -9 ]
[.17549 .00222 3 0 ]
[.32717 .00222 -3 -9 ]
[.32717 .00222 3 0 ]
[.47885 .00222 -3 -9 ]
[.47885 .00222 3 0 ]
[.63053 .00222 -3 -9 ]
[.63053 .00222 3 0 ]
[.78221 .00222 -3 -9 ]
[.78221 .00222 3 0 ]
[.93389 .00222 -3 -9 ]
[.93389 .00222 3 0 ]
[.01131 .11787 -6 -4.5 ]
[.01131 .11787 0 4.5 ]
[.01131 .22102 -6 -4.5 ]
[.01131 .22102 0 4.5 ]
[.01131 .32417 -6 -4.5 ]
[.01131 .32417 0 4.5 ]
[.01131 .42732 -6 -4.5 ]
[.01131 .42732 0 4.5 ]
[.01131 .53047 -6 -4.5 ]
[.01131 .53047 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.17549 .01472 m
.17549 .02097 L
s
[(1)] .17549 .00222 0 1 Mshowa
.32717 .01472 m
.32717 .02097 L
s
[(2)] .32717 .00222 0 1 Mshowa
.47885 .01472 m
.47885 .02097 L
s
[(3)] .47885 .00222 0 1 Mshowa
.63053 .01472 m
.63053 .02097 L
s
[(4)] .63053 .00222 0 1 Mshowa
.78221 .01472 m
.78221 .02097 L
s
[(5)] .78221 .00222 0 1 Mshowa
.93389 .01472 m
.93389 .02097 L
s
[(6)] .93389 .00222 0 1 Mshowa
.125 Mabswid
.05415 .01472 m
.05415 .01847 L
s
.08448 .01472 m
.08448 .01847 L
s
.11482 .01472 m
.11482 .01847 L
s
.14515 .01472 m
.14515 .01847 L
s
.20583 .01472 m
.20583 .01847 L
s
.23616 .01472 m
.23616 .01847 L
s
.2665 .01472 m
.2665 .01847 L
s
.29683 .01472 m
.29683 .01847 L
s
.35751 .01472 m
.35751 .01847 L
s
.38784 .01472 m
.38784 .01847 L
s
.41818 .01472 m
.41818 .01847 L
s
.44851 .01472 m
.44851 .01847 L
s
.50919 .01472 m
.50919 .01847 L
s
.53952 .01472 m
.53952 .01847 L
s
.56986 .01472 m
.56986 .01847 L
s
.60019 .01472 m
.60019 .01847 L
s
.66087 .01472 m
.66087 .01847 L
s
.6912 .01472 m
.6912 .01847 L
s
.72154 .01472 m
.72154 .01847 L
s
.75187 .01472 m
.75187 .01847 L
s
.81255 .01472 m
.81255 .01847 L
s
.84288 .01472 m
.84288 .01847 L
s
.87322 .01472 m
.87322 .01847 L
s
.90355 .01472 m
.90355 .01847 L
s
.96423 .01472 m
.96423 .01847 L
s
.99456 .01472 m
.99456 .01847 L
s
.25 Mabswid
0 .01472 m
1 .01472 L
s
.02381 .11787 m
.03006 .11787 L
s
[(1)] .01131 .11787 1 0 Mshowa
.02381 .22102 m
.03006 .22102 L
s
[(2)] .01131 .22102 1 0 Mshowa
.02381 .32417 m
.03006 .32417 L
s
[(3)] .01131 .32417 1 0 Mshowa
.02381 .42732 m
.03006 .42732 L
s
[(4)] .01131 .42732 1 0 Mshowa
.02381 .53047 m
.03006 .53047 L
s
[(5)] .01131 .53047 1 0 Mshowa
.125 Mabswid
.02381 .03535 m
.02756 .03535 L
s
.02381 .05598 m
.02756 .05598 L
s
.02381 .07661 m
.02756 .07661 L
s
.02381 .09724 m
.02756 .09724 L
s
.02381 .1385 m
.02756 .1385 L
s
.02381 .15913 m
.02756 .15913 L
s
.02381 .17976 m
.02756 .17976 L
s
.02381 .20039 m
.02756 .20039 L
s
.02381 .24165 m
.02756 .24165 L
s
.02381 .26228 m
.02756 .26228 L
s
.02381 .28291 m
.02756 .28291 L
s
.02381 .30354 m
.02756 .30354 L
s
.02381 .3448 m
.02756 .3448 L
s
.02381 .36543 m
.02756 .36543 L
s
.02381 .38606 m
.02756 .38606 L
s
.02381 .40669 m
.02756 .40669 L
s
.02381 .44795 m
.02756 .44795 L
s
.02381 .46858 m
.02756 .46858 L
s
.02381 .48921 m
.02756 .48921 L
s
.02381 .50984 m
.02756 .50984 L
s
.02381 .5511 m
.02756 .5511 L
s
.02381 .57173 m
.02756 .57173 L
s
.02381 .59236 m
.02756 .59236 L
s
.02381 .61299 m
.02756 .61299 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.008 w
.87276 .17067 Mdot
.32721 .24942 Mdot
.81967 .1658 Mdot
.20601 .18947 Mdot
.26173 .19123 Mdot
.92407 .17554 Mdot
.55249 .17559 Mdot
.1 .18014 Mdot
.76449 .16264 Mdot
.74324 .09577 Mdot
.60842 .12563 Mdot
.15186 .18512 Mdot
.47293 .17529 Mdot
.42352 .1802 Mdot
.73545 .09575 Mdot
.05049 .1752 Mdot
.71201 .16266 Mdot
.76794 .16275 Mdot
.47662 .17481 Mdot
.10315 .18044 Mdot
.60379 .17073 Mdot
.31746 .18951 Mdot
.19853 .24981 Mdot
.37164 .18518 Mdot
.8706 .12637 Mdot
.65685 .16585 Mdot
.14859 .18481 Mdot
.81631 .16553 Mdot
.46938 .17572 Mdot
.61778 .12196 Mdot
.86955 .17036 Mdot
.33551 .24728 Mdot
.20261 .18926 Mdot
.70513 .1629 Mdot
.41721 .1808 Mdot
.04285 .17414 Mdot
.09685 .17985 Mdot
.76105 .16255 Mdot
.75104 .09604 Mdot
.36837 .18549 Mdot
.88027 .13034 Mdot
.6535 .16613 Mdot
.70857 .16278 Mdot
.42037 .1805 Mdot
.04677 .17471 Mdot
.25828 .19123 Mdot
.92088 .17524 Mdot
.54928 .1759 Mdot
.6006 .17104 Mdot
.31405 .18971 Mdot
.2067 .25167 Mdot
.18185 .24537 Mdot
.61021 .17011 Mdot
.26864 .19121 Mdot
.9305 .17615 Mdot
.66356 .16533 Mdot
.37817 .18456 Mdot
.32426 .18908 Mdot
.48759 .18728 Mdot
.21284 .18986 Mdot
.47477 .19224 Mdot
.31092 .25299 Mdot
.55884 .175 Mdot
.87914 .17129 Mdot
.57858 .13815 Mdot
.16171 .18603 Mdot
.77481 .16301 Mdot
.48467 .1736 Mdot
.45526 .20064 Mdot
.50059 .18319 Mdot
.82968 .16664 Mdot
.05754 .17603 Mdot
.7189 .16248 Mdot
.85202 .11918 Mdot
.71983 .09645 Mdot
.10947 .18104 Mdot
.42982 .17961 Mdot
.92727 .17584 Mdot
.19025 .24771 Mdot
.55567 .1753 Mdot
.26519 .19123 Mdot
.32086 .1893 Mdot
.6602 .16559 Mdot
.607 .17042 Mdot
.15513 .18543 Mdot
.82302 .16607 Mdot
.59881 .12954 Mdot
.87595 .17098 Mdot
.31901 .25133 Mdot
.20942 .18967 Mdot
.77138 .16288 Mdot
.4805 .17426 Mdot
.10631 .18074 Mdot
.15842 .18573 Mdot
.82635 .16635 Mdot
.58888 .13371 Mdot
.86119 .12266 Mdot
.05407 .17564 Mdot
.37491 .18487 Mdot
.42667 .1799 Mdot
.72765 .09597 Mdot
.71545 .16257 Mdot
.0874 .17897 Mdot
.75069 .16234 Mdot
.40771 .18171 Mdot
.77464 .09836 Mdot
.64465 .11237 Mdot
.13885 .18388 Mdot
.45921 .17682 Mdot
.64021 .16728 Mdot
.35516 .18668 Mdot
.09563 .21252 Mdot
.0283 .17137 Mdot
.69483 .16335 Mdot
.92364 .14924 Mdot
.59105 .17196 Mdot
.30379 .19024 Mdot
.91138 .17435 Mdot
.23072 .25577 Mdot
.08431 .2075 Mdot
.64352 .16698 Mdot
.96886 .18134 Mdot
.19244 .18857 Mdot
.85988 .16942 Mdot
.8062 .16478 Mdot
.53947 .17688 Mdot
.24791 .19112 Mdot
.3612 .23939 Mdot
.78262 .09962 Mdot
.08423 .17867 Mdot
.6914 .16352 Mdot
.40453 .18202 Mdot
.45593 .17715 Mdot
.65327 .10964 Mdot
.74723 .1623 Mdot
.08863 .10818 Mdot
.94072 .25635 Mdot
.93004 .25184 Mdot
.95033 .26089 Mdot
.10393 .11239 Mdot
.09741 .11082 Mdot
.90822 .17405 Mdot
.2386 .25664 Mdot
.24446 .19105 Mdot
.30036 .19039 Mdot
.96386 .18035 Mdot
.58789 .17227 Mdot
.80281 .16455 Mdot
.18907 .18832 Mdot
.13562 .18357 Mdot
.37008 .23629 Mdot
.53612 .17723 Mdot
.85663 .16911 Mdot
.62692 .11853 Mdot
.14534 .1845 Mdot
.7576 .16247 Mdot
.46592 .17611 Mdot
.19922 .18904 Mdot
.86634 .17005 Mdot
.81295 .16528 Mdot
.03863 .17346 Mdot
.70169 .16304 Mdot
.90065 .13905 Mdot
.75887 .09657 Mdot
.09371 .17956 Mdot
.41405 .1811 Mdot
.03229 .18581 Mdot
.05497 .19458 Mdot
.31064 .1899 Mdot
.36508 .18579 Mdot
.89025 .13455 Mdot
.65017 .16641 Mdot
.54604 .17621 Mdot
.25482 .1912 Mdot
.34393 .24489 Mdot
.21478 .25328 Mdot
.59741 .17135 Mdot
.91771 .17494 Mdot
.46254 .17647 Mdot
.63587 .11533 Mdot
.09055 .17926 Mdot
.75414 .1624 Mdot
.80958 .16502 Mdot
.19583 .18881 Mdot
.14209 .18419 Mdot
.91165 .1439 Mdot
.03391 .17259 Mdot
.35847 .18639 Mdot
.41089 .1814 Mdot
.76673 .09734 Mdot
.69826 .16319 Mdot
.30721 .19007 Mdot
.97508 .18279 Mdot
.59423 .17166 Mdot
.64684 .16669 Mdot
.36178 .18609 Mdot
.07164 .20187 Mdot
.35249 .24226 Mdot
.54278 .17654 Mdot
.86311 .16974 Mdot
.91454 .17464 Mdot
.22278 .25465 Mdot
.25136 .19117 Mdot
.18166 .18774 Mdot
.84946 .16843 Mdot
.12854 .18288 Mdot
.79533 .16406 Mdot
.52843 .17812 Mdot
.23685 .19087 Mdot
.39035 .22861 Mdot
.80047 .10328 Mdot
.07722 .17801 Mdot
.39751 .18269 Mdot
.44883 .17783 Mdot
.67178 .10448 Mdot
.73962 .16226 Mdot
.68729 .16374 Mdot
.4007 .18239 Mdot
.63294 .16794 Mdot
.34782 .18731 Mdot
.08041 .17831 Mdot
.74307 .16227 Mdot
.79233 .10148 Mdot
.25585 .25769 Mdot
.58091 .17293 Mdot
.90129 .17341 Mdot
.95466 .17889 Mdot
.1184 .22233 Mdot
.29278 .19069 Mdot
.67999 .10253 Mdot
.12534 .18257 Mdot
.73616 .16226 Mdot
.44563 .17813 Mdot
.17831 .18747 Mdot
.84618 .16812 Mdot
.79192 .16386 Mdot
.34113 .18785 Mdot
.68048 .16415 Mdot
.6264 .16855 Mdot
.80872 .10533 Mdot
.07402 .17771 Mdot
.39431 .18301 Mdot
.9509 .17839 Mdot
.12812 .22634 Mdot
.28934 .1908 Mdot
.34448 .18758 Mdot
.68388 .16394 Mdot
.62967 .16825 Mdot
.52477 .17858 Mdot
.23341 .19076 Mdot
.3999 .22475 Mdot
.26364 .25776 Mdot
.57776 .17323 Mdot
.89814 .17311 Mdot
.90759 .174 Mdot
.24022 .25679 Mdot
.5354 .17731 Mdot
.24374 .19104 Mdot
.29965 .19042 Mdot
.96295 .1802 Mdot
.58722 .17233 Mdot
.13496 .1835 Mdot
.80211 .1645 Mdot
.65502 .10911 Mdot
.85597 .16905 Mdot
.37194 .23562 Mdot
.18837 .18826 Mdot
.08673 .1789 Mdot
.74997 .16233 Mdot
.40706 .18177 Mdot
.77629 .0986 Mdot
.64644 .11178 Mdot
.13818 .18381 Mdot
.45854 .17688 Mdot
.63952 .16734 Mdot
.35448 .18674 Mdot
.09784 .21349 Mdot
.0269 .17102 Mdot
.69412 .16338 Mdot
.92639 .15045 Mdot
.2403 .19096 Mdot
.90444 .1737 Mdot
.38104 .23223 Mdot
.53196 .1777 Mdot
.58407 .17263 Mdot
.29622 .19056 Mdot
.24804 .25736 Mdot
.66346 .10667 Mdot
.13174 .18319 Mdot
.45204 .17752 Mdot
.18501 .18801 Mdot
.85272 .16874 Mdot
.79872 .16428 Mdot
.40388 .18208 Mdot
.78427 .09991 Mdot
.6907 .16356 Mdot
.74652 .16229 Mdot
.45527 .17721 Mdot
.08357 .17861 Mdot
.10833 .21805 Mdot
.63623 .16764 Mdot
.95864 .17948 Mdot
.94167 .15718 Mdot
.97114 .16856 Mdot
.35115 .18703 Mdot
.67031 .16483 Mdot
.38465 .18394 Mdot
.61666 .16949 Mdot
.33103 .18861 Mdot
.06427 .17675 Mdot
.7258 .16235 Mdot
.83424 .11293 Mdot
.29496 .25557 Mdot
.56516 .17441 Mdot
.8855 .1719 Mdot
.93705 .17681 Mdot
.16465 .23997 Mdot
.27555 .19113 Mdot
.43069 .21156 Mdot
.51237 .18053 Mdot
.16832 .18662 Mdot
.83631 .16722 Mdot
.88867 .17221 Mdot
.28708 .25649 Mdot
.2231 .19036 Mdot
.70408 .09814 Mdot
.1158 .18165 Mdot
.43612 .17902 Mdot
.49466 .17164 Mdot
.55596 .14815 Mdot
.78167 .16332 Mdot
.32765 .18885 Mdot
.66693 .16507 Mdot
.17332 .24279 Mdot
.61343 .1698 Mdot
.84304 .11594 Mdot
.06094 .1764 Mdot
.38141 .18425 Mdot
.88233 .1716 Mdot
.30291 .2544 Mdot
.21625 .19004 Mdot
.2721 .19118 Mdot
.93375 .17648 Mdot
.56201 .1747 Mdot
.16501 .18633 Mdot
.833 .16693 Mdot
.56771 .14293 Mdot
.50719 .18158 Mdot
.21968 .19021 Mdot
.44221 .20645 Mdot
.43297 .17932 Mdot
.71198 .09717 Mdot
.72235 .16241 Mdot
.77824 .16316 Mdot
.48927 .17277 Mdot
.11263 .18134 Mdot
.0708 .1774 Mdot
.73271 .16228 Mdot
.3911 .18332 Mdot
.81709 .10763 Mdot
.6881 .10082 Mdot
.12215 .18226 Mdot
.44245 .17843 Mdot
.94381 .17754 Mdot
.14679 .23362 Mdot
.28245 .19099 Mdot
.33777 .18811 Mdot
.67708 .16437 Mdot
.62315 .16886 Mdot
.27143 .25758 Mdot
.57462 .17353 Mdot
.89498 .17281 Mdot
.9473 .17794 Mdot
.13757 .2301 Mdot
.2859 .1909 Mdot
.17497 .18719 Mdot
.8429 .16782 Mdot
.78851 .16367 Mdot
.52094 .17912 Mdot
.22997 .19064 Mdot
.40974 .22064 Mdot
.82559 .11016 Mdot
.06755 .17708 Mdot
.67369 .16459 Mdot
.38788 .18363 Mdot
.43929 .17873 Mdot
.69613 .09936 Mdot
.72926 .16231 Mdot
.279 .19107 Mdot
.9404 .17716 Mdot
.56832 .17412 Mdot
.6199 .16917 Mdot
.33441 .18837 Mdot
.15581 .23692 Mdot
.51685 .17975 Mdot
.22654 .19051 Mdot
.41996 .21626 Mdot
.27924 .25716 Mdot
.57147 .17382 Mdot
.89183 .17251 Mdot
.78509 .16349 Mdot
.5018 .16982 Mdot
.11897 .18195 Mdot
.17164 .18691 Mdot
.83961 .16752 Mdot
.54236 .15418 Mdot
.4285 .18861 Mdot
.66544 .11043 Mdot
.10701 .18789 Mdot
.74769 .15924 Mdot
.79984 .16097 Mdot
.42925 .20276 Mdot
.14922 .18932 Mdot
.87867 .13594 Mdot
.08615 .19461 Mdot
.34335 .19134 Mdot
.39053 .18846 Mdot
.77411 .10228 Mdot
.69513 .16012 Mdot
.60786 .16513 Mdot
.30283 .19348 Mdot
.89173 .16563 Mdot
.23217 .25197 Mdot
.12662 .21973 Mdot
.65473 .16231 Mdot
.91475 .15973 Mdot
.19817 .19238 Mdot
.84902 .16402 Mdot
.57393 .14909 Mdot
.57069 .16467 Mdot
.25023 .19422 Mdot
.34084 .24142 Mdot
.68792 .10494 Mdot
.13979 .18878 Mdot
.73681 .1592 Mdot
.42201 .18806 Mdot
.43607 .19733 Mdot
.58714 .14161 Mdot
.78924 .16043 Mdot
.08973 .19891 Mdot
.68447 .1606 Mdot
.89485 .14419 Mdot
.79657 .10653 Mdot
.0999 .1882 Mdot
.3813 .18895 Mdot
.59909 .16547 Mdot
.29206 .19385 Mdot
.25454 .25372 Mdot
.14666 .22843 Mdot
.64462 .16296 Mdot
.91361 .16231 Mdot
.83926 .1634 Mdot
.36299 .23402 Mdot
.18774 .19178 Mdot
.23937 .19404 Mdot
.88376 .1656 Mdot
.5658 .1635 Mdot
.76938 .15967 Mdot
.43728 .19133 Mdot
.72939 .09983 Mdot
.1232 .18804 Mdot
.16885 .19058 Mdot
.82063 .16219 Mdot
.62162 .1257 Mdot
.36348 .19004 Mdot
.83789 .11888 Mdot
.66503 .16167 Mdot
.7167 .15944 Mdot
.40786 .18788 Mdot
.08959 .18988 Mdot
.18726 .24259 Mdot
.62645 .16412 Mdot
.27199 .19424 Mdot
.90546 .16482 Mdot
.90505 .15026 Mdot
.09663 .2039 Mdot
.3241 .19248 Mdot
.56257 .15914 Mdot
.21941 .19341 Mdot
.4008 .21819 Mdot
.29593 .25145 Mdot
.58409 .16561 Mdot
.86772 .16506 Mdot
.43377 .18964 Mdot
.64316 .1175 Mdot
.11484 .18786 Mdot
.75856 .1594 Mdot
.81031 .16156 Mdot
.41728 .20987 Mdot
.15892 .18993 Mdot
.85922 .12723 Mdot
.08648 .19173 Mdot
.35351 .19069 Mdot
.3994 .18809 Mdot
.75173 .10001 Mdot
.70588 .15973 Mdot
.20973 .2482 Mdot
.617 .16467 Mdot
.89904 .1654 Mdot
.91226 .15582 Mdot
.10947 .21128 Mdot
.31352 .19302 Mdot
.20874 .19293 Mdot
.85852 .16458 Mdot
.56582 .15496 Mdot
.57692 .16533 Mdot
.26111 .19429 Mdot
.31838 .24734 Mdot
.75694 .15936 Mdot
.43308 .18945 Mdot
.75506 .10022 Mdot
.11364 .18785 Mdot
.15746 .18984 Mdot
.80876 .16147 Mdot
.64646 .11636 Mdot
.35201 .19079 Mdot
.86226 .12852 Mdot
.65322 .1624 Mdot
.70428 .15978 Mdot
.3981 .18814 Mdot
.08623 .19208 Mdot
.21308 .24889 Mdot
.61562 .16475 Mdot
.25948 .19429 Mdot
.898 .16546 Mdot
.9129 .1565 Mdot
.1118 .21249 Mdot
.31193 .19309 Mdot
.56671 .1542 Mdot
.20716 .19285 Mdot
.41937 .20872 Mdot
.32173 .24657 Mdot
.57593 .16525 Mdot
.85712 .1645 Mdot
.4276 .1885 Mdot
.66879 .1095 Mdot
.10591 .18792 Mdot
.74607 .15922 Mdot
.79826 .16088 Mdot
.43058 .20184 Mdot
.1478 .18924 Mdot
.88132 .13722 Mdot
.08641 .19515 Mdot
.34182 .19144 Mdot
.38917 .18853 Mdot
.77745 .10279 Mdot
.69353 .16018 Mdot
.23551 .25236 Mdot
.60653 .16519 Mdot
.89058 .16564 Mdot
.91477 .16018 Mdot
.12946 .22103 Mdot
.30123 .19354 Mdot
.19661 .19229 Mdot
.84758 .16393 Mdot
.5756 .14806 Mdot
.56987 .16453 Mdot
.24861 .1942 Mdot
.34417 .2404 Mdot
.13068 .18833 Mdot
.77855 .15998 Mdot
.41464 .18785 Mdot
.71036 .10132 Mdot
.60455 .13311 Mdot
.17746 .19113 Mdot
.43825 .19353 Mdot
.67391 .16115 Mdot
.37178 .18952 Mdot
.09823 .20491 Mdot
.09376 .18889 Mdot
.72594 .15928 Mdot
.81902 .11257 Mdot
.28123 .19411 Mdot
.90995 .16395 Mdot
.59075 .16564 Mdot
.63471 .1636 Mdot
.33301 .19198 Mdot
.16829 .23657 Mdot
.38423 .22562 Mdot
.56286 .16159 Mdot
.82928 .16275 Mdot
.87527 .16537 Mdot
.27687 .25338 Mdot
.22856 .19374 Mdot
.76777 .15962 Mdot
.43689 .19103 Mdot
.73272 .09972 Mdot
.12193 .188 Mdot
.16736 .19048 Mdot
.8191 .1621 Mdot
.62475 .12443 Mdot
.36201 .19014 Mdot
.84114 .12007 Mdot
.66348 .16176 Mdot
.71508 .15947 Mdot
.40663 .1879 Mdot
.08899 .1901 Mdot
.90457 .16493 Mdot
.1906 .24353 Mdot
.27037 .19426 Mdot
.32253 .19257 Mdot
.90644 .15119 Mdot
.62503 .16421 Mdot
.56278 .15862 Mdot
.21782 .19334 Mdot
.40348 .21691 Mdot
.29926 .25096 Mdot
.58297 .16559 Mdot
.86637 .165 Mdot
.9537 .26938 Mdot
.07447 .09576 Mdot
.96067 .27167 Mdot
.0859 .1009 Mdot
.06277 .09059 Mdot
.9331 .2608 Mdot
.94428 .26563 Mdot
.10729 .11135 Mdot
.96642 .27178 Mdot
.96507 .26984 Mdot
.96494 .27244 Mdot
.09574 .1054 Mdot
.10305 .10894 Mdot
.03445 .08237 Mdot
.89255 .24147 Mdot
.03476 .08116 Mdot
.89546 .24341 Mdot
.89292 .24068 Mdot
.0422 .08821 Mdot
.03697 .08479 Mdot
.92134 .25556 Mdot
.04375 .08298 Mdot
.05221 .08614 Mdot
.03788 .08132 Mdot
.90154 .24649 Mdot
.91042 .25062 Mdot
.04379 .08913 Mdot
.8977 .2413 Mdot
.03795 .08551 Mdot
.89347 .24067 Mdot
.90467 .24306 Mdot
.0618 .09796 Mdot
.05193 .0934 Mdot
.89985 .24567 Mdot
.03444 .08133 Mdot
.0369 .08115 Mdot
.03479 .08284 Mdot
.89235 .24118 Mdot
.89449 .24286 Mdot
.10079 .1117 Mdot
.95424 .26292 Mdot
.94541 .2585 Mdot
.96094 .2668 Mdot
.10828 .11262 Mdot
.10604 .11273 Mdot
.07257 .10242 Mdot
.92425 .2496 Mdot
.91379 .24589 Mdot
.93512 .25392 Mdot
.09298 .10955 Mdot
.08329 .10639 Mdot
.03901 .08157 Mdot
.90339 .24737 Mdot
.04551 .08359 Mdot
.91283 .25172 Mdot
.89661 .24404 Mdot
.03427 .08197 Mdot
.03524 .08107 Mdot
.94668 .26663 Mdot
.06546 .09176 Mdot
.07722 .09699 Mdot
.05452 .08709 Mdot
.92407 .25679 Mdot
.93582 .262 Mdot
.05701 .09581 Mdot
.9093 .24444 Mdot
.9011 .2421 Mdot
.91924 .24776 Mdot
.07834 .10461 Mdot
.06748 .10037 Mdot
.03613 .08413 Mdot
.89254 .24076 Mdot
.89294 .24182 Mdot
.89535 .24086 Mdot
.04786 .09134 Mdot
.04074 .08733 Mdot
.93259 .25287 Mdot
.09085 .10889 Mdot
.92173 .24866 Mdot
.08084 .10552 Mdot
.09917 .11129 Mdot
.95234 .26192 Mdot
.9431 .25743 Mdot
.04984 .09236 Mdot
.90282 .24255 Mdot
.89645 .24105 Mdot
.9115 .24514 Mdot
.07002 .10141 Mdot
.05938 .09689 Mdot
.96554 .27241 Mdot
.09771 .10633 Mdot
.10433 .1096 Mdot
.08839 .10203 Mdot
.95556 .27005 Mdot
.96191 .27198 Mdot
.95959 .26596 Mdot
.10804 .11275 Mdot
.10506 .11259 Mdot
.10782 .11175 Mdot
.96635 .27143 Mdot
.96434 .26922 Mdot
.93558 .49751 Mdot
.91428 .58031 Mdot
.92841 .51163 Mdot
.91314 .56903 Mdot
.918 .58814 Mdot
.95575 .46883 Mdot
.94462 .48355 Mdot
.68605 .16052 Mdot
.3827 .18887 Mdot
.0889 .19816 Mdot
.92165 .53125 Mdot
.91664 .5491 Mdot
.07682 .47759 Mdot
.96889 .566 Mdot
.06877 .40743 Mdot
.95399 .58078 Mdot
.05873 .41139 Mdot
.07654 .40847 Mdot
.04424 .53362 Mdot
.03085 .54951 Mdot
.96902 .45299 Mdot
.9314 .59424 Mdot
.92375 .59297 Mdot
.94131 .59059 Mdot
.04589 .42153 Mdot
.03093 .4365 Mdot
.78014 .16004 Mdot
.43819 .194 Mdot
.70702 .10173 Mdot
.13201 .18839 Mdot
.17897 .19123 Mdot
.83078 .16285 Mdot
.60175 .13441 Mdot
.37322 .18943 Mdot
.81569 .11157 Mdot
.67547 .16107 Mdot
.72755 .15926 Mdot
.41578 .18786 Mdot
.0946 .18875 Mdot
.91063 .16376 Mdot
.16501 .23542 Mdot
.28285 .19408 Mdot
.33456 .19188 Mdot
.89272 .14303 Mdot
.63616 .16351 Mdot
.56315 .16193 Mdot
.23017 .19379 Mdot
.38117 .22691 Mdot
.27355 .25356 Mdot
.59196 .16562 Mdot
.87657 .16542 Mdot
.96563 .2704 Mdot
.10659 .11089 Mdot
.96215 .2676 Mdot
.10834 .11243 Mdot
.1016 .10821 Mdot
.96419 .27239 Mdot
.96632 .27205 Mdot
.07064 .40729 Mdot
.03395 .54595 Mdot
.97224 .56243 Mdot
.95602 .26389 Mdot
.10684 .11281 Mdot
.10226 .11204 Mdot
.93034 .25959 Mdot
.05001 .08527 Mdot
.06015 .08946 Mdot
.04214 .08245 Mdot
.90812 .24957 Mdot
.91865 .25436 Mdot
.95927 .27127 Mdot
.08332 .09973 Mdot
.09364 .10443 Mdot
.0717 .09452 Mdot
.94179 .26458 Mdot
.95169 .26862 Mdot
.92346 .52521 Mdot
.91508 .55548 Mdot
.08451 .45059 Mdot
.07847 .47217 Mdot
.91311 .57187 Mdot
.93738 .49449 Mdot
.92984 .50844 Mdot
.05548 .51884 Mdot
.08623 .42079 Mdot
.08238 .41313 Mdot
.08754 .43185 Mdot
.07189 .49079 Mdot
.06463 .50486 Mdot
.03427 .43298 Mdot
.94383 .58898 Mdot
.93335 .59392 Mdot
.95712 .57787 Mdot
.06115 .41002 Mdot
.04893 .41881 Mdot
.94687 .48042 Mdot
.9191 .58946 Mdot
.9149 .58228 Mdot
.92525 .59359 Mdot
.97218 .4494 Mdot
.95847 .46547 Mdot
.50506 .28168 Mdot
.42481 .10369 Mdot
.5221 .27409 Mdot
.44707 .09399 Mdot
.40615 .11132 Mdot
.48404 .2886 Mdot
.49273 .28631 Mdot
.50318 .06882 Mdot
.56669 .25371 Mdot
.58709 .24515 Mdot
.54371 .26401 Mdot
.4698 .08349 Mdot
.48896 .07467 Mdot
.37602 .12596 Mdot
.48731 .27649 Mdot
.374 .12534 Mdot
.47958 .28184 Mdot
.49901 .26981 Mdot
.39239 .12371 Mdot
.38216 .12548 Mdot
.47831 .28911 Mdot
.38195 .12092 Mdot
.39193 .11692 Mdot
.376 .12367 Mdot
.47574 .28572 Mdot
.47545 .28813 Mdot
.47615 .09597 Mdot
.60062 .23146 Mdot
.45692 .10396 Mdot
.5846 .23564 Mdot
.61311 .22886 Mdot
.5071 .08069 Mdot
.49324 .08799 Mdot
.52679 .25652 Mdot
.41823 .11722 Mdot
.40183 .1216 Mdot
.43707 .11124 Mdot
.56599 .24144 Mdot
.54618 .24858 Mdot
.52435 .06477 Mdot
.61495 .23411 Mdot
.6226 .23085 Mdot
.60321 .23874 Mdot
.5133 .06562 Mdot
.52025 .06441 Mdot
.51712 .07455 Mdot
.62599 .22766 Mdot
.62161 .22766 Mdot
.62629 .22874 Mdot
.52545 .06657 Mdot
.52317 .06982 Mdot
.44009 .0971 Mdot
.50088 .28337 Mdot
.46314 .0866 Mdot
.5164 .27671 Mdot
.48976 .28722 Mdot
.40138 .11321 Mdot
.41871 .10623 Mdot
.58134 .2475 Mdot
.49936 .07028 Mdot
.51061 .06635 Mdot
.48369 .07704 Mdot
.53682 .26721 Mdot
.55986 .2567 Mdot
.95141 .5628 Mdot
.03787 .44635 Mdot
.94951 .56224 Mdot
.03415 .45054 Mdot
.04172 .44277 Mdot
.95767 .55983 Mdot
.95421 .56198 Mdot
.38847 .11828 Mdot
.47713 .28897 Mdot
.48201 .28892 Mdot
.94866 .5604 Mdot
.03092 .45489 Mdot
.37973 .12187 Mdot
.58977 .23421 Mdot
.4816 .09352 Mdot
.57182 .23952 Mdot
.46287 .10159 Mdot
.49781 .08569 Mdot
.61611 .22836 Mdot
.6048 .23051 Mdot
.40646 .12045 Mdot
.53251 .25408 Mdot
.51435 .26215 Mdot
.5522 .24631 Mdot
.44304 .10915 Mdot
.42372 .11557 Mdot
.62071 .23171 Mdot
.52341 .0645 Mdot
.52545 .06587 Mdot
.51845 .0646 Mdot
.59881 .24047 Mdot
.61184 .23535 Mdot
.62337 .22754 Mdot
.51936 .07297 Mdot
.51055 .0787 Mdot
.52424 .06868 Mdot
.62559 .22927 Mdot
.6265 .22788 Mdot
.76973 .20514 Mdot
.91223 .07209 Mdot
.74075 .20427 Mdot
.85446 .04914 Mdot
.95648 .08998 Mdot
.83341 .2126 Mdot
.80021 .20774 Mdot
.52708 .08742 Mdot
.68171 .20722 Mdot
.64898 .21169 Mdot
.71188 .20491 Mdot
.63522 .04446 Mdot
.57228 .06912 Mdot
.19606 .14523 Mdot
.08196 .27877 Mdot
.16242 .14001 Mdot
.0384 .26121 Mdot
.13628 .30053 Mdot
.25595 .14919 Mdot
.22685 .1481 Mdot
.86992 .22039 Mdot
.08584 .12026 Mdot
.04494 .10603 Mdot
.12546 .13177 Mdot
.95006 .24537 Mdot
.90926 .23147 Mdot
.21411 .14713 Mdot
.11072 .29041 Mdot
.18223 .14335 Mdot
.06298 .27107 Mdot
.19311 .32068 Mdot
.27252 .14913 Mdot
.24376 .14892 Mdot
.89235 .22639 Mdot
.10895 .12728 Mdot
.06861 .11452 Mdot
.14718 .13692 Mdot
.97384 .25432 Mdot
.93271 .2392 Mdot
.40452 .13023 Mdot
.57398 .2296 Mdot
.53331 .24313 Mdot
.61297 .21901 Mdot
.48542 .10371 Mdot
.44443 .11826 Mdot
.30179 .14779 Mdot
.44909 .27596 Mdot
.39766 .29669 Mdot
.49204 .25872 Mdot
.36708 .13897 Mdot
.33295 .14461 Mdot
.49603 .28519 Mdot
.41133 .10924 Mdot
.50968 .27971 Mdot
.43125 .10095 Mdot
.39576 .11541 Mdot
.47974 .28912 Mdot
.48634 .28811 Mdot
.49376 .07258 Mdot
.55072 .2608 Mdot
.97413 .09695 Mdot
.52823 .27122 Mdot
.45409 .09079 Mdot
.4761 .08055 Mdot
.48455 .27826 Mdot
.37986 .12575 Mdot
.37498 .12589 Mdot
.38887 .12439 Mdot
.50936 .26454 Mdot
.49506 .27195 Mdot
.476 .28858 Mdot
.37738 .12296 Mdot
.38454 .11985 Mdot
.37418 .12494 Mdot
.47802 .28317 Mdot
.47529 .2866 Mdot
.55062 .2371 Mdot
.42117 .12554 Mdot
.50956 .25193 Mdot
.38253 .13568 Mdot
.4617 .11236 Mdot
.6286 .21553 Mdot
.59076 .22472 Mdot
.28476 .14876 Mdot
.42142 .28719 Mdot
.35167 .31395 Mdot
.46769 .26843 Mdot
.34698 .14257 Mdot
.31467 .1467 Mdot
.7529 .20445 Mdot
.8822 .05999 Mdot
.71706 .02243 Mdot
.93161 .07996 Mdot
.81385 .20948 Mdot
.78235 .20601 Mdot
.6632 .20949 Mdot
.5454 .08002 Mdot
.50289 .09701 Mdot
.59476 .06007 Mdot
.72416 .20445 Mdot
.69471 .20602 Mdot
.08168 .46896 Mdot
.92251 .52184 Mdot
.07441 .48895 Mdot
.08678 .45016 Mdot
.92944 .50607 Mdot
.91155 .57652 Mdot
.91175 .56245 Mdot
.08375 .41189 Mdot
.05748 .51745 Mdot
.0461 .53229 Mdot
.06682 .50334 Mdot
.08949 .43054 Mdot
.08783 .41951 Mdot
.0375 .42902 Mdot
.9462 .58808 Mdot
.97483 .44573 Mdot
.93486 .59462 Mdot
.96048 .57527 Mdot
.06405 .40766 Mdot
.05219 .41532 Mdot
.91439 .58587 Mdot
.94799 .47765 Mdot
.93777 .492 Mdot
.96035 .4623 Mdot
.92611 .59543 Mdot
.91934 .59219 Mdot
.93439 .49734 Mdot
.91303 .5827 Mdot
.9267 .51165 Mdot
.91118 .57194 Mdot
.91724 .59012 Mdot
.95538 .46829 Mdot
.94386 .48321 Mdot
.03346 .46526 Mdot
.96028 .54134 Mdot
.96418 .53821 Mdot
.95647 .54515 Mdot
.91336 .55368 Mdot
.03035 .46647 Mdot
.05989 .4098 Mdot
.9702 .56526 Mdot
.95474 .58079 Mdot
.03253 .54835 Mdot
.07779 .40721 Mdot
.06998 .40602 Mdot
.96909 .45215 Mdot
.93126 .59551 Mdot
.92332 .5946 Mdot
.94153 .59135 Mdot
.04688 .41993 Mdot
.0315 .43524 Mdot
.08705 .41742 Mdot
.06477 .50668 Mdot
.0825 .4105 Mdot
.05496 .52089 Mdot
.07274 .49251 Mdot
.08821 .44343 Mdot
.08937 .42752 Mdot
.96643 .56922 Mdot
.06781 .4064 Mdot
.05707 .41162 Mdot
.07611 .40657 Mdot
.04307 .536 Mdot
.029 .55234 Mdot
.918 .59095 Mdot
.95721 .46607 Mdot
.91351 .58394 Mdot
.94538 .48114 Mdot
.97122 .44976 Mdot
.93257 .59527 Mdot
.92433 .59497 Mdot
.92081 .52677 Mdot
.91257 .5575 Mdot
.07962 .47551 Mdot
.91126 .57374 Mdot
.93563 .49534 Mdot
.92771 .50953 Mdot
.0699 .49799 Mdot
.08924 .43616 Mdot
.06127 .51203 Mdot
.08879 .42323 Mdot
.92507 .51533 Mdot
.9112 .56866 Mdot
.07698 .48284 Mdot
.07319 .40599 Mdot
.03796 .54208 Mdot
.97619 .55881 Mdot
.05069 .52651 Mdot
.08554 .41445 Mdot
.08027 .40863 Mdot
.92918 .59569 Mdot
.02774 .43925 Mdot
.96563 .45611 Mdot
.04336 .42323 Mdot
.95133 .58387 Mdot
.93883 .59289 Mdot
.91235 .58048 Mdot
.94143 .48662 Mdot
.93241 .50068 Mdot
.95244 .47195 Mdot
.92171 .59382 Mdot
.91606 .58862 Mdot
.94907 .5617 Mdot
.03288 .45215 Mdot
.94864 .55945 Mdot
.02992 .45645 Mdot
.03646 .44784 Mdot
.95309 .56244 Mdot
.9506 .56275 Mdot
.02718 .46561 Mdot
.95117 .55237 Mdot
.95395 .5482 Mdot
.94935 .55623 Mdot
.02787 .46034 Mdot
.02693 .46349 Mdot
.05151 .43979 Mdot
.97055 .54546 Mdot
.0499 .43884 Mdot
.96753 .54978 Mdot
.97267 .54153 Mdot
.05139 .44512 Mdot
.05202 .44196 Mdot
.95634 .56076 Mdot
.04406 .44102 Mdot
.04031 .44399 Mdot
.04735 .43924 Mdot
.96392 .55407 Mdot
.96006 .55787 Mdot
.05193 .44302 Mdot
.97319 .54025 Mdot
.05183 .44046 Mdot
.97145 .54395 Mdot
.97378 .53738 Mdot
.04875 .45046 Mdot
.05088 .44646 Mdot
.96149 .55655 Mdot
.04839 .43893 Mdot
.04534 .44021 Mdot
.05061 .43903 Mdot
.96873 .54817 Mdot
.9653 .55252 Mdot
.03439 .46474 Mdot
.96131 .54043 Mdot
.96518 .53753 Mdot
.95744 .5441 Mdot
.02862 .46657 Mdot
.0311 .46627 Mdot
.04573 .45465 Mdot
.97141 .535 Mdot
.97317 .53558 Mdot
.96865 .53567 Mdot
.03818 .46213 Mdot
.04209 .45866 Mdot
.04906 .43882 Mdot
.96625 .55139 Mdot
.04621 .43974 Mdot
.96251 .55555 Mdot
.96953 .54703 Mdot
.05197 .44103 Mdot
.05104 .43929 Mdot
.95207 .56272 Mdot
.03889 .44533 Mdot
.03509 .4494 Mdot
.04271 .44199 Mdot
.95866 .55906 Mdot
.95507 .56154 Mdot
.04482 .45574 Mdot
.97077 .53506 Mdot
.97281 .53531 Mdot
.9678 .53605 Mdot
.03716 .46291 Mdot
.04107 .45964 Mdot
.05176 .44386 Mdot
.97346 .5394 Mdot
.97201 .54291 Mdot
.97373 .5368 Mdot
.04803 .45156 Mdot
.05042 .44748 Mdot
.96652 .53672 Mdot
.03575 .4639 Mdot
.96977 .53528 Mdot
.03963 .46093 Mdot
.03223 .46584 Mdot
.95884 .54268 Mdot
.96276 .53926 Mdot
.04965 .44895 Mdot
.97353 .53611 Mdot
.9737 .53832 Mdot
.97218 .53507 Mdot
.04348 .45722 Mdot
.04693 .45311 Mdot
.94877 .55836 Mdot
.02905 .45795 Mdot
.02739 .46161 Mdot
.0317 .45376 Mdot
.94992 .56252 Mdot
.94878 .561 Mdot
.95517 .54666 Mdot
.02758 .46611 Mdot
.02942 .4666 Mdot
.02688 .4644 Mdot
.9499 .55487 Mdot
.95209 .55085 Mdot
.57261 .21415 Mdot
.40509 .14513 Mdot
.50403 .23438 Mdot
.34575 .15682 Mdot
.46958 .12753 Mdot
.69089 .19234 Mdot
.63334 .20031 Mdot
.23312 .16249 Mdot
.13322 .27323 Mdot
.97033 .23348 Mdot
.38056 .27676 Mdot
.28898 .16259 Mdot
.27128 .3008 Mdot
.19544 .15916 Mdot
.04472 .24244 Mdot
.13702 .14921 Mdot
.92364 .21939 Mdot
.441 .25565 Mdot
.3077 .16133 Mdot
.25175 .16317 Mdot
.74686 .19028 Mdot
.95263 .10994 Mdot
.55521 .0994 Mdot
.07462 .13342 Mdot
.86145 .20382 Mdot
.80313 .19408 Mdot
.27035 .16321 Mdot
.47612 .24355 Mdot
.21438 .16115 Mdot
.0787 .25409 Mdot
.55114 .22007 Mdot
.38493 .14967 Mdot
.32662 .15941 Mdot
.82227 .19667 Mdot
.09609 .13934 Mdot
.02752 .11912 Mdot
.15682 .15318 Mdot
.94609 .22596 Mdot
.8816 .20836 Mdot
.05202 .12676 Mdot
.84169 .19991 Mdot
.78423 .19216 Mdot
.90227 .21355 Mdot
.17628 .1565 Mdot
.11682 .14459 Mdot
.44712 .1341 Mdot
.67196 .19433 Mdot
.61354 .20428 Mdot
.72828 .19032 Mdot
.91753 .09785 Mdot
.52124 .11105 Mdot
.1168 .14459 Mdot
.90225 .21354 Mdot
.05199 .12675 Mdot
.84167 .19991 Mdot
.9703 .23347 Mdot
.2331 .16249 Mdot
.17626 .1565 Mdot
.67195 .19433 Mdot
.52122 .11105 Mdot
.4471 .13411 Mdot
.91748 .09783 Mdot
.78421 .19216 Mdot
.72826 .19032 Mdot
.63331 .20031 Mdot
.46954 .12754 Mdot
.57258 .21416 Mdot
.40506 .14513 Mdot
.55516 .09942 Mdot
.74684 .19028 Mdot
.69086 .19234 Mdot
.27149 .30079 Mdot
.38042 .2768 Mdot
.13312 .2732 Mdot
.504 .23439 Mdot
.34573 .15683 Mdot
.28895 .16259 Mdot
.74786 .0527 Mdot
.85703 .07672 Mdot
.70963 .191 Mdot
.6097 .08027 Mdot
.02751 .11911 Mdot
.82226 .19667 Mdot
.76548 .1909 Mdot
.36517 .15358 Mdot
.59333 .2089 Mdot
.52853 .22673 Mdot
.65279 .19699 Mdot
.49381 .12001 Mdot
.42574 .13994 Mdot
.4761 .24356 Mdot
.3266 .15941 Mdot
.27034 .16321 Mdot
.38492 .14968 Mdot
.61353 .20429 Mdot
.55113 .22007 Mdot
.8816 .20836 Mdot
.15681 .15318 Mdot
.09608 .13933 Mdot
.21436 .16115 Mdot
.07868 .25409 Mdot
.94608 .22596 Mdot
.3065 .19011 Mdot
.97368 .18245 Mdot
.22445 .2549 Mdot
.59356 .17172 Mdot
.64614 .16675 Mdot
.36109 .18615 Mdot
.0744 .2031 Mdot
.86244 .16967 Mdot
.35431 .24168 Mdot
.19512 .18875 Mdot
.25064 .19116 Mdot
.91389 .17459 Mdot
.54207 .17662 Mdot
.6288 .11784 Mdot
.14465 .18444 Mdot
.75687 .16245 Mdot
.46522 .17618 Mdot
.1985 .18899 Mdot
.86566 .16998 Mdot
.81225 .16522 Mdot
.03765 .17328 Mdot
.70097 .16307 Mdot
.90294 .14005 Mdot
.76052 .09671 Mdot
.09303 .17949 Mdot
.4134 .18116 Mdot
.59039 .17203 Mdot
.30308 .19027 Mdot
.91074 .17429 Mdot
.23236 .25597 Mdot
.08668 .20856 Mdot
.64283 .16705 Mdot
.9678 .18112 Mdot
.19174 .18851 Mdot
.85921 .16936 Mdot
.8055 .16473 Mdot
.53876 .17695 Mdot
.24719 .1911 Mdot
.36304 .23877 Mdot
.75342 .16239 Mdot
.46185 .17654 Mdot
.08988 .17919 Mdot
.14141 .18413 Mdot
.80888 .16497 Mdot
.63771 .11469 Mdot
.91409 .14498 Mdot
.03279 .17235 Mdot
.35779 .18645 Mdot
.41023 .18146 Mdot
.76838 .09753 Mdot
.69754 .16322 Mdot
.92363 .21939 Mdot
.19544 .15916 Mdot
.86145 .20381 Mdot
.13702 .14921 Mdot
.25175 .16317 Mdot
.44101 .25564 Mdot
.04472 .24244 Mdot
.5552 .0994 Mdot
.74686 .19028 Mdot
.69089 .19234 Mdot
.80313 .19408 Mdot
.07462 .13342 Mdot
.95263 .10994 Mdot
.70964 .191 Mdot
.60973 .08026 Mdot
.49381 .12001 Mdot
.85708 .07674 Mdot
.76549 .1909 Mdot
.7478 .0527 Mdot
.52853 .22673 Mdot
.36518 .15358 Mdot
.30771 .16133 Mdot
.42575 .13994 Mdot
.65279 .19699 Mdot
.59334 .2089 Mdot
.05916 .19639 Mdot
.64946 .16647 Mdot
.30992 .18993 Mdot
.02938 .18489 Mdot
.89244 .13549 Mdot
.04195 .174 Mdot
.36439 .18585 Mdot
.54534 .17629 Mdot
.25409 .1912 Mdot
.34573 .24435 Mdot
.21647 .25359 Mdot
.59673 .17142 Mdot
.91705 .17488 Mdot
.34575 .15683 Mdot
.5726 .21415 Mdot
.50402 .23438 Mdot
.63333 .20031 Mdot
.46957 .12753 Mdot
.40508 .14513 Mdot
.23311 .16249 Mdot
.13317 .27321 Mdot
.97032 .23347 Mdot
.3805 .27677 Mdot
.28897 .1626 Mdot
.27137 .30079 Mdot
.24816 .14905 Mdot
.29736 .329 Mdot
.21873 .14752 Mdot
.11928 .29383 Mdot
.40673 .29308 Mdot
.30638 .14744 Mdot
.27691 .14903 Mdot
.93893 .24138 Mdot
.15273 .1381 Mdot
.11493 .12896 Mdot
.18726 .14408 Mdot
.06967 .27378 Mdot
.0271 .25676 Mdot
.61868 .21768 Mdot
.49171 .10133 Mdot
.58008 .22777 Mdot
.45064 .11618 Mdot
.53361 .0848 Mdot
.68646 .20674 Mdot
.65418 .21084 Mdot
.33794 .14393 Mdot
.49838 .25624 Mdot
.4559 .2732 Mdot
.53957 .24091 Mdot
.41048 .12861 Mdot
.37258 .13785 Mdot
.63952 .21337 Mdot
.51561 .09201 Mdot
.60261 .22156 Mdot
.47422 .10787 Mdot
.55925 .07441 Mdot
.70387 .20536 Mdot
.67311 .20819 Mdot
.35746 .14079 Mdot
.52213 .2472 Mdot
.48065 .26323 Mdot
.56302 .23302 Mdot
.4334 .12181 Mdot
.39402 .13294 Mdot
.82408 .21101 Mdot
.03376 .10183 Mdot
.94495 .08536 Mdot
.07481 .11663 Mdot
.89838 .22815 Mdot
.85969 .21796 Mdot
.73296 .20429 Mdot
.83006 .04019 Mdot
.61395 .05252 Mdot
.89881 .06666 Mdot
.79171 .20684 Mdot
.76177 .20475 Mdot
.59676 .22309 Mdot
.468 .11012 Mdot
.55688 .23501 Mdot
.42732 .1237 Mdot
.50928 .09451 Mdot
.66823 .20881 Mdot
.63415 .2144 Mdot
.31944 .14622 Mdot
.47425 .26579 Mdot
.42916 .28405 Mdot
.5159 .24953 Mdot
.38829 .13434 Mdot
.35222 .14171 Mdot
.78704 .2064 Mdot
.93839 .08271 Mdot
.89082 .06344 Mdot
.02749 .09943 Mdot
.85407 .2167 Mdot
.81897 .21022 Mdot
.69935 .20566 Mdot
.604 .05641 Mdot
.5523 .07723 Mdot
.81027 .03364 Mdot
.75736 .20458 Mdot
.7286 .20436 Mdot
.96291 .09254 Mdot
.83878 .21358 Mdot
.91942 .07501 Mdot
.80509 .20832 Mdot
.87579 .22188 Mdot
.09202 .12222 Mdot
.05124 .10835 Mdot
.71634 .20471 Mdot
.65073 .03894 Mdot
.58002 .06599 Mdot
.86542 .05337 Mdot
.77426 .20542 Mdot
.74514 .2043 Mdot
.23139 .14837 Mdot
.14742 .30481 Mdot
.08918 .2817 Mdot
.36686 .3085 Mdot
.28924 .14856 Mdot
.26036 .14922 Mdot
.13133 .13324 Mdot
.95636 .24769 Mdot
.91545 .23344 Mdot
.04484 .26377 Mdot
.20094 .14581 Mdot
.16779 .14099 Mdot
.55107 .22009 Mdot
.38487 .14969 Mdot
.47602 .24358 Mdot
.32655 .15941 Mdot
.44705 .13412 Mdot
.67189 .19434 Mdot
.61347 .2043 Mdot
.15676 .15317 Mdot
.94603 .22594 Mdot
.88155 .20835 Mdot
.07859 .25406 Mdot
.27029 .16321 Mdot
.21432 .16115 Mdot
.1762 .15649 Mdot
.97023 .23345 Mdot
.11674 .14457 Mdot
.90219 .21353 Mdot
.13286 .27311 Mdot
.27201 .30077 Mdot
.23304 .16248 Mdot
.72821 .19032 Mdot
.91736 .09779 Mdot
.52114 .11108 Mdot
.05192 .12673 Mdot
.84162 .1999 Mdot
.78415 .19216 Mdot
.25172 .16317 Mdot
.44092 .25568 Mdot
.19541 .15915 Mdot
.04467 .24243 Mdot
.52848 .22675 Mdot
.36514 .15358 Mdot
.30767 .16133 Mdot
.8031 .19408 Mdot
.07458 .13341 Mdot
.95258 .10992 Mdot
.13699 .1492 Mdot
.9236 .21938 Mdot
.86142 .20381 Mdot
.8568 .07664 Mdot
.76544 .19089 Mdot
.74823 .05271 Mdot
.82222 .19666 Mdot
.09603 .13932 Mdot
.02745 .11909 Mdot
.4257 .13995 Mdot
.65275 .197 Mdot
.59329 .20891 Mdot
.70959 .19101 Mdot
.60952 .08033 Mdot
.49376 .12003 Mdot
.02734 .11906 Mdot
.82214 .19665 Mdot
.85637 .07649 Mdot
.76537 .19089 Mdot
.88147 .20833 Mdot
.15668 .15316 Mdot
.09595 .1393 Mdot
.65267 .19701 Mdot
.49366 .12006 Mdot
.42562 .13997 Mdot
.6092 .08044 Mdot
.7489 .05274 Mdot
.70952 .19101 Mdot
.38478 .1497 Mdot
.37915 .30388 Mdot
.55097 .22012 Mdot
.43648 .28108 Mdot
.3242 .14569 Mdot
.29368 .14832 Mdot
.21424 .16114 Mdot
.07842 .254 Mdot
.94594 .22592 Mdot
.47589 .24363 Mdot
.32647 .15942 Mdot
.27021 .16321 Mdot
.6908 .19235 Mdot
.55503 .09946 Mdot
.63324 .20033 Mdot
.46947 .12756 Mdot
.95248 .10988 Mdot
.80304 .19407 Mdot
.74677 .19028 Mdot
.28889 .1626 Mdot
.50392 .23442 Mdot
.38009 .27692 Mdot
.57251 .21418 Mdot
.40499 .14515 Mdot
.34567 .15684 Mdot
.44077 .25573 Mdot
.3076 .16134 Mdot
.25165 .16317 Mdot
.36506 .1536 Mdot
.59321 .20893 Mdot
.52839 .22677 Mdot
.86135 .2038 Mdot
.13691 .14918 Mdot
.07451 .13339 Mdot
.19534 .15914 Mdot
.04456 .24239 Mdot
.92353 .21936 Mdot
.32125 .21733 Mdot
.72547 .1192 Mdot
.20277 .21907 Mdot
.73489 .14303 Mdot
.76171 .1427 Mdot
.31482 .22469 Mdot
.20909 .21379 Mdot
.78823 .12734 Mdot
.21433 .22693 Mdot
.28963 .21095 Mdot
.3109 .21288 Mdot
.76103 .12025 Mdot
.70872 .14243 Mdot
.6834 .139 Mdot
.28063 .21066 Mdot
.79702 .13717 Mdot
.25525 .23468 Mdot
.22278 .22974 Mdot
.70056 .14191 Mdot
.79385 .13025 Mdot
.22802 .21129 Mdot
.78429 .1411 Mdot
.69481 .12512 Mdot
.67994 .13317 Mdot
.25358 .21049 Mdot
.29027 .2323 Mdot
.22217 .19995 Mdot
.83599 .15378 Mdot
.35836 .22532 Mdot
.62409 .13837 Mdot
.62156 .1507 Mdot
.26658 .20035 Mdot
.29276 .24424 Mdot
.7559 .15321 Mdot
.38296 .20729 Mdot
.15058 .2012 Mdot
.1815 .19935 Mdot
.79906 .15393 Mdot
.67269 .11691 Mdot
.67922 .13558 Mdot
.26319 .21045 Mdot
.68646 .15377 Mdot
.24282 .23367 Mdot
.68821 .14031 Mdot
.79799 .13506 Mdot
.2164 .24165 Mdot
.6478 .15403 Mdot
.85817 .14915 Mdot
.84702 .13441 Mdot
.15683 .22088 Mdot
.31058 .19979 Mdot
.76905 .1424 Mdot
.30929 .22714 Mdot
.71532 .12037 Mdot
.21365 .21281 Mdot
.23512 .21093 Mdot
.78935 .14023 Mdot
.68847 .12754 Mdot
.31538 .21388 Mdot
.75083 .11913 Mdot
.71609 .14272 Mdot
.74295 .14303 Mdot
.32131 .21924 Mdot
.20296 .21718 Mdot
.21556 .2274 Mdot
.70734 .14236 Mdot
.2882 .2109 Mdot
.7893 .12781 Mdot
.76293 .12053 Mdot
.20292 .21946 Mdot
.30992 .21272 Mdot
.67918 .13522 Mdot
.26162 .21045 Mdot
.28047 .23377 Mdot
.24479 .23389 Mdot
.68733 .14012 Mdot
.79798 .13543 Mdot
.31926 .22184 Mdot
.70425 .12248 Mdot
.2051 .21529 Mdot
.75248 .14293 Mdot
.77715 .14187 Mdot
.30069 .22992 Mdot
.22031 .21192 Mdot
.7722 .12231 Mdot
.20466 .22165 Mdot
.30433 .21198 Mdot
.31925 .21541 Mdot
.73814 .11866 Mdot
.72535 .14294 Mdot
.23309 .23215 Mdot
.69335 .14116 Mdot
.7971 .13299 Mdot
.78176 .12493 Mdot
.20893 .22448 Mdot
.29668 .21133 Mdot
.24415 .21064 Mdot
.79405 .13889 Mdot
.68301 .13045 Mdot
.6803 .1373 Mdot
.27123 .2105 Mdot
.268 .23472 Mdot
.37036 .21916 Mdot
.63319 .1327 Mdot
.17141 .19946 Mdot
.78727 .15371 Mdot
.82656 .15408 Mdot
.31365 .24035 Mdot
.21011 .19972 Mdot
.76897 .10921 Mdot
.14511 .20277 Mdot
.35959 .19971 Mdot
.38169 .20438 Mdot
.69263 .1119 Mdot
.74327 .15314 Mdot
.6748 .15398 Mdot
.33958 .19935 Mdot
.85494 .13984 Mdot
.16994 .22716 Mdot
.14139 .20831 Mdot
.71764 .15325 Mdot
.80989 .11858 Mdot
.25391 .20034 Mdot
.85424 .15122 Mdot
.61833 .1484 Mdot
.63866 .15364 Mdot
.29843 .20001 Mdot
.23757 .24501 Mdot
.65432 .12313 Mdot
.19263 .19942 Mdot
.76876 .15338 Mdot
.38087 .2113 Mdot
.34229 .23187 Mdot
.61896 .14322 Mdot
.81068 .1541 Mdot
.1412 .20655 Mdot
.72411 .15319 Mdot
.80013 .11567 Mdot
.72491 .10753 Mdot
.158 .20018 Mdot
.37496 .2015 Mdot
.65816 .15414 Mdot
.3227 .19956 Mdot
.19591 .2365 Mdot
.14748 .21505 Mdot
.69886 .15354 Mdot
.83468 .12808 Mdot
.84452 .15312 Mdot
.27052 .24622 Mdot
.23487 .20016 Mdot
.27956 .20027 Mdot
.85947 .14615 Mdot
.62719 .15231 Mdot
.82136 .15413 Mdot
.32388 .23775 Mdot
.63944 .12951 Mdot
.20406 .19961 Mdot
.24742 .2003 Mdot
.85139 .15201 Mdot
.61771 .14689 Mdot
.37998 .20321 Mdot
.70339 .10993 Mdot
.73676 .15313 Mdot
.78106 .15359 Mdot
.37498 .21625 Mdot
.16658 .19962 Mdot
.81905 .12171 Mdot
.14248 .21033 Mdot
.33401 .19939 Mdot
.3679 .20036 Mdot
.74734 .10728 Mdot
.71121 .15333 Mdot
.63441 .1533 Mdot
.29207 .20012 Mdot
.2487 .24597 Mdot
.17803 .23045 Mdot
.66899 .15407 Mdot
.85744 .14225 Mdot
.35075 .22861 Mdot
.621 .14093 Mdot
.18698 .19937 Mdot
.80494 .15402 Mdot
.84042 .1535 Mdot
.2817 .2455 Mdot
.22849 .20006 Mdot
.73612 .10712 Mdot
.15409 .20062 Mdot
.37164 .20086 Mdot
.38238 .20916 Mdot
.66318 .11993 Mdot
.76236 .15329 Mdot
.15162 .21785 Mdot
.69262 .15365 Mdot
.84136 .13132 Mdot
.77985 .11098 Mdot
.14312 .20384 Mdot
.35488 .19952 Mdot
.27308 .20033 Mdot
.85921 .14777 Mdot
.62412 .15159 Mdot
.65287 .15411 Mdot
.3167 .19967 Mdot
.20594 .23925 Mdot
.79126 .12876 Mdot
.21818 .22833 Mdot
.68567 .13972 Mdot
.28531 .21079 Mdot
.30786 .2124 Mdot
.76663 .12117 Mdot
.70465 .1422 Mdot
.28436 .23326 Mdot
.67928 .13446 Mdot
.7875 .14059 Mdot
.79778 .13614 Mdot
.24879 .23428 Mdot
.2585 .21046 Mdot
.22415 .21157 Mdot
.78095 .1415 Mdot
.31728 .22328 Mdot
.69917 .12379 Mdot
.68113 .13188 Mdot
.24896 .21055 Mdot
.29554 .23121 Mdot
.73022 .143 Mdot
.32053 .21634 Mdot
.20339 .22028 Mdot
.20694 .21447 Mdot
.75727 .14283 Mdot
.73159 .11881 Mdot
.23676 .2328 Mdot
.69125 .14086 Mdot
.26812 .21047 Mdot
.79764 .13382 Mdot
.78444 .12585 Mdot
.21083 .22543 Mdot
.29401 .21117 Mdot
.68452 .12952 Mdot
.24117 .21072 Mdot
.3037 .22905 Mdot
.27211 .23452 Mdot
.6797 .13667 Mdot
.7927 .13937 Mdot
.21177 .21316 Mdot
.76626 .14253 Mdot
.71917 .11984 Mdot
.69074 .1266 Mdot
.23231 .21105 Mdot
.31162 .22619 Mdot
.31377 .21347 Mdot
.75485 .11949 Mdot
.71319 .14262 Mdot
.73983 .14304 Mdot
.32147 .21846 Mdot
.2027 .21788 Mdot
.3129 .21327 Mdot
.75683 .1197 Mdot
.21187 .22591 Mdot
.71176 .14257 Mdot
.73827 .14304 Mdot
.32147 .21809 Mdot
.20266 .21824 Mdot
.79781 .13422 Mdot
.23864 .2331 Mdot
.26657 .21046 Mdot
.29265 .2111 Mdot
.7857 .12631 Mdot
.69025 .14069 Mdot
.67942 .13406 Mdot
.25694 .21046 Mdot
.78652 .14076 Mdot
.28626 .23298 Mdot
.25081 .23443 Mdot
.68491 .1395 Mdot
.7976 .13647 Mdot
.21088 .21335 Mdot
.76484 .14259 Mdot
.72114 .11961 Mdot
.69197 .12613 Mdot
.23094 .21112 Mdot
.3127 .22572 Mdot
.77708 .12353 Mdot
.20643 .223 Mdot
.69692 .14158 Mdot
.30073 .21163 Mdot
.31758 .21462 Mdot
.74435 .11876 Mdot
.72079 .14285 Mdot
.26145 .23483 Mdot
.68166 .13822 Mdot
.79581 .13806 Mdot
.79573 .13162 Mdot
.22759 .23098 Mdot
.27609 .21056 Mdot
.68537 .12905 Mdot
.23969 .21076 Mdot
.30513 .2286 Mdot
.27414 .23438 Mdot
.67949 .13634 Mdot
.79196 .13959 Mdot
.74787 .14299 Mdot
.32055 .22053 Mdot
.20381 .21616 Mdot
.21692 .21232 Mdot
.77331 .14215 Mdot
.70946 .12137 Mdot
.24266 .21068 Mdot
.79339 .13913 Mdot
.30222 .22949 Mdot
.68373 .12999 Mdot
.67997 .13699 Mdot
.26968 .21048 Mdot
.27006 .23463 Mdot
.75095 .14295 Mdot
.31975 .2214 Mdot
.20462 .21557 Mdot
.21915 .21205 Mdot
.77589 .14197 Mdot
.70594 .12209 Mdot
.75285 .1193 Mdot
.2028 .21752 Mdot
.71463 .14268 Mdot
.31459 .21367 Mdot
.32142 .21885 Mdot
.71723 .1201 Mdot
.74139 .14304 Mdot
.23491 .23248 Mdot
.69228 .14101 Mdot
.7974 .13341 Mdot
.78313 .12538 Mdot
.20985 .22496 Mdot
.29536 .21125 Mdot
.7832 .14124 Mdot
.29206 .23195 Mdot
.69621 .12467 Mdot
.22671 .21138 Mdot
.25203 .21051 Mdot
.79667 .13748 Mdot
.68027 .13275 Mdot
.32106 .21699 Mdot
.7275 .11904 Mdot
.73333 .14303 Mdot
.76024 .14275 Mdot
.3157 .22421 Mdot
.20833 .21401 Mdot
.77387 .12271 Mdot
.20518 .22209 Mdot
.30316 .21185 Mdot
.31874 .21514 Mdot
.74021 .11867 Mdot
.72382 .14291 Mdot
.68277 .13875 Mdot
.27913 .21062 Mdot
.25731 .23476 Mdot
.22434 .23016 Mdot
.69932 .14181 Mdot
.79455 .13071 Mdot
.68219 .13849 Mdot
.27762 .21059 Mdot
.79626 .13777 Mdot
.25938 .23481 Mdot
.22594 .23058 Mdot
.69811 .1417 Mdot
.79517 .13117 Mdot
.22542 .21147 Mdot
.78209 .14138 Mdot
.69767 .12422 Mdot
.68067 .13232 Mdot
.25049 .21053 Mdot
.29382 .23159 Mdot
.74942 .14297 Mdot
.32018 .22096 Mdot
.20419 .21586 Mdot
.21802 .21218 Mdot
.77461 .14206 Mdot
.70768 .12172 Mdot
.7755 .12311 Mdot
.20578 .22254 Mdot
.30196 .21174 Mdot
.31818 .21487 Mdot
.74228 .1187 Mdot
.7223 .14288 Mdot
.26006 .21045 Mdot
.79791 .13579 Mdot
.28243 .23353 Mdot
.6792 .13484 Mdot
.68648 .13992 Mdot
.28677 .21084 Mdot
.24678 .2341 Mdot
.76766 .14247 Mdot
.31048 .22666 Mdot
.21269 .21298 Mdot
.23371 .21099 Mdot
.78844 .14042 Mdot
.68958 .12707 Mdot
.32082 .21666 Mdot
.72954 .11892 Mdot
.73178 .14301 Mdot
.75876 .14279 Mdot
.31652 .22375 Mdot
.20761 .21424 Mdot
.21684 .22787 Mdot
.70598 .14228 Mdot
.79031 .12829 Mdot
.7648 .12084 Mdot
.20313 .21987 Mdot
.30891 .21256 Mdot
.31735 .19966 Mdot
.84068 .13097 Mdot
.20485 .23897 Mdot
.65343 .15412 Mdot
.69329 .15364 Mdot
.35539 .19954 Mdot
.15113 .21754 Mdot
.84087 .15347 Mdot
.2805 .24561 Mdot
.22917 .20007 Mdot
.27378 .20032 Mdot
.85927 .14761 Mdot
.62443 .15167 Mdot
.63872 .12986 Mdot
.2047 .19962 Mdot
.78173 .1536 Mdot
.37452 .21655 Mdot
.3228 .23804 Mdot
.61775 .14707 Mdot
.82192 .15413 Mdot
.14331 .20372 Mdot
.73746 .15313 Mdot
.77869 .11077 Mdot
.70223 .11012 Mdot
.16709 .1996 Mdot
.38019 .20333 Mdot
.8546 .13956 Mdot
.16913 .22681 Mdot
.63914 .15367 Mdot
.2991 .2 Mdot
.34017 .19935 Mdot
.80887 .11825 Mdot
.67543 .15397 Mdot
.31252 .24061 Mdot
.61845 .14855 Mdot
.8271 .15407 Mdot
.85451 .15112 Mdot
.23638 .24488 Mdot
.25461 .20035 Mdot
.6622 .12027 Mdot
.18758 .19937 Mdot
.38226 .20938 Mdot
.34987 .22896 Mdot
.62074 .14119 Mdot
.80556 .15403 Mdot
.71834 .15324 Mdot
.37201 .20092 Mdot
.14133 .20811 Mdot
.1545 .20057 Mdot
.76304 .15329 Mdot
.73492 .10714 Mdot
.68712 .15376 Mdot
.35044 .19942 Mdot
.84645 .13409 Mdot
.15623 .22054 Mdot
.1419 .20498 Mdot
.73095 .15315 Mdot
.78932 .11294 Mdot
.26728 .20035 Mdot
.85831 .14901 Mdot
.62182 .1508 Mdot
.64834 .15404 Mdot
.31125 .19977 Mdot
.21526 .24141 Mdot
.3326 .23516 Mdot
.61786 .14533 Mdot
.19876 .19951 Mdot
.81654 .15414 Mdot
.84844 .15258 Mdot
.2587 .24636 Mdot
.24165 .20024 Mdot
.71319 .10858 Mdot
.1625 .19983 Mdot
.37793 .20234 Mdot
.37817 .21386 Mdot
.64584 .12659 Mdot
.77545 .15348 Mdot
.33461 .19939 Mdot
.81809 .12137 Mdot
.17713 .2301 Mdot
.66962 .15406 Mdot
.71191 .15332 Mdot
.36831 .20041 Mdot
.14232 .2101 Mdot
.85171 .15193 Mdot
.2475 .2459 Mdot
.24812 .20031 Mdot
.29276 .20011 Mdot
.85721 .142 Mdot
.63485 .15334 Mdot
.35757 .22567 Mdot
.62372 .13866 Mdot
.79969 .15394 Mdot
.83647 .15376 Mdot
.29157 .2444 Mdot
.22285 .19996 Mdot
.15095 .20113 Mdot
.7566 .15322 Mdot
.74614 .10724 Mdot
.67164 .11723 Mdot
.18208 .19935 Mdot
.38293 .20748 Mdot
.63112 .15293 Mdot
.2867 .20019 Mdot
.84862 .15255 Mdot
.2581 .24635 Mdot
.18544 .23314 Mdot
.66425 .15412 Mdot
.85875 .14405 Mdot
.19907 .19952 Mdot
.81683 .15414 Mdot
.64545 .12677 Mdot
.61784 .14543 Mdot
.24199 .20025 Mdot
.3321 .23532 Mdot
.38276 .20592 Mdot
.6811 .11459 Mdot
.14799 .2018 Mdot
.75046 .15317 Mdot
.79402 .15384 Mdot
.364 .2226 Mdot
.17705 .19937 Mdot
.82617 .12444 Mdot
.14416 .21224 Mdot
.32919 .19946 Mdot
.36446 .20003 Mdot
.75672 .10786 Mdot
.70586 .15341 Mdot
.26762 .20035 Mdot
.85838 .14894 Mdot
.29099 .24447 Mdot
.62195 .15085 Mdot
.6486 .15404 Mdot
.31157 .19977 Mdot
.2147 .24129 Mdot
.80001 .15394 Mdot
.35718 .22585 Mdot
.18237 .19935 Mdot
.22318 .19997 Mdot
.83671 .15374 Mdot
.62353 .1388 Mdot
.37806 .20239 Mdot
.71261 .10865 Mdot
.7313 .15314 Mdot
.77579 .15349 Mdot
.378 .214 Mdot
.16274 .19982 Mdot
.15593 .22038 Mdot
.68745 .15375 Mdot
.84617 .13393 Mdot
.78876 .11282 Mdot
.14196 .20491 Mdot
.35071 .19942 Mdot
.19487 .23619 Mdot
.65874 .15414 Mdot
.28025 .20027 Mdot
.85944 .14596 Mdot
.8339 .12773 Mdot
.14711 .21476 Mdot
.32333 .19955 Mdot
.6188 .14345 Mdot
.23556 .20017 Mdot
.34134 .23221 Mdot
.26932 .24626 Mdot
.62755 .15237 Mdot
.84493 .15307 Mdot
.17195 .19945 Mdot
.78793 .15372 Mdot
.6915 .11214 Mdot
.63259 .13303 Mdot
.21077 .19973 Mdot
.3698 .21948 Mdot
.36008 .19974 Mdot
.76779 .10905 Mdot
.69954 .15352 Mdot
.74397 .15314 Mdot
.38183 .20452 Mdot
.14536 .20266 Mdot
.64374 .1539 Mdot
.30538 .19989 Mdot
.85676 .15013 Mdot
.22542 .24332 Mdot
.16203 .22354 Mdot
.68138 .15387 Mdot
.85093 .13686 Mdot
.21693 .19985 Mdot
.83204 .15395 Mdot
.62752 .13602 Mdot
.61987 .1498 Mdot
.26111 .20036 Mdot
.30188 .24278 Mdot
.76944 .15338 Mdot
.38064 .21155 Mdot
.15845 .20014 Mdot
.19325 .19943 Mdot
.81128 .1541 Mdot
.65342 .12348 Mdot
.79905 .11537 Mdot
.14124 .20638 Mdot
.34555 .19936 Mdot
.37528 .20158 Mdot
.72372 .10761 Mdot
.7248 .15318 Mdot
.75625 .15322 Mdot
.38295 .20739 Mdot
.74674 .10726 Mdot
.15076 .20117 Mdot
.18179 .19935 Mdot
.79938 .15393 Mdot
.67216 .11707 Mdot
.33431 .19939 Mdot
.81857 .12154 Mdot
.66931 .15406 Mdot
.71156 .15333 Mdot
.36811 .20038 Mdot
.1424 .21021 Mdot
.21583 .24153 Mdot
.64807 .15403 Mdot
.26693 .20035 Mdot
.85824 .14908 Mdot
.84674 .13425 Mdot
.15653 .22071 Mdot
.31092 .19978 Mdot
.6239 .13851 Mdot
.22251 .19996 Mdot
.35796 .2255 Mdot
.29216 .24432 Mdot
.62169 .15075 Mdot
.83623 .15377 Mdot
.38009 .20327 Mdot
.70281 .11002 Mdot
.14321 .20378 Mdot
.73711 .15313 Mdot
.7814 .15359 Mdot
.37475 .2164 Mdot
.16684 .19961 Mdot
.84102 .13114 Mdot
.15137 .2177 Mdot
.31702 .19966 Mdot
.35514 .19953 Mdot
.77926 .11087 Mdot
.69295 .15365 Mdot
.24809 .24594 Mdot
.63463 .15332 Mdot
.85155 .15197 Mdot
.85733 .14212 Mdot
.17758 .23027 Mdot
.29242 .20011 Mdot
.20438 .19961 Mdot
.82164 .15413 Mdot
.63907 .12969 Mdot
.61773 .14698 Mdot
.24777 .2003 Mdot
.32334 .2379 Mdot
.17649 .19938 Mdot
.79338 .15382 Mdot
.38271 .20576 Mdot
.68218 .11431 Mdot
.62801 .13572 Mdot
.21627 .19984 Mdot
.36468 .22226 Mdot
.70518 .15343 Mdot
.36402 .19999 Mdot
.14442 .2125 Mdot
.14768 .20188 Mdot
.74977 .15317 Mdot
.75791 .10796 Mdot
.30471 .1999 Mdot
.85138 .13716 Mdot
.64323 .15388 Mdot
.68074 .15388 Mdot
.34498 .19936 Mdot
.16274 .22389 Mdot
.30304 .24257 Mdot
.61968 .14968 Mdot
.83152 .15396 Mdot
.85655 .15025 Mdot
.22658 .24352 Mdot
.26042 .20036 Mdot
.77512 .15348 Mdot
.37834 .21372 Mdot
.71378 .10851 Mdot
.16226 .19984 Mdot
.19844 .19951 Mdot
.81625 .15414 Mdot
.64625 .12642 Mdot
.35018 .19941 Mdot
.78988 .11307 Mdot
.68679 .15377 Mdot
.7306 .15315 Mdot
.37779 .2023 Mdot
.14184 .20505 Mdot
.85887 .14427 Mdot
.18641 .23347 Mdot
.28601 .2002 Mdot
.32858 .19946 Mdot
.82704 .12479 Mdot
.66365 .15412 Mdot
.61789 .14523 Mdot
.2413 .20024 Mdot
.33312 .23499 Mdot
.2593 .24637 Mdot
.63071 .15288 Mdot
.84825 .15262 Mdot
.14122 .20647 Mdot
.72446 .15319 Mdot
.34527 .19936 Mdot
.79958 .11552 Mdot
.72431 .10757 Mdot
.15823 .20016 Mdot
.37512 .20154 Mdot
.64348 .15389 Mdot
.30505 .19989 Mdot
.226 .24342 Mdot
.16238 .22372 Mdot
.68106 .15388 Mdot
.85116 .13701 Mdot
.84473 .1531 Mdot
.26992 .24624 Mdot
.61888 .14333 Mdot
.23522 .20016 Mdot
.27991 .20027 Mdot
.85946 .14606 Mdot
.62737 .15234 Mdot
.38076 .21143 Mdot
.65387 .12331 Mdot
.76911 .15338 Mdot
.81098 .1541 Mdot
.34182 .23204 Mdot
.19294 .19943 Mdot
.70552 .15342 Mdot
.36424 .20001 Mdot
.82661 .12462 Mdot
.14429 .21237 Mdot
.14783 .20184 Mdot
.75012 .15317 Mdot
.75731 .10791 Mdot
.28636 .2002 Mdot
.85881 .14416 Mdot
.63091 .15291 Mdot
.66395 .15412 Mdot
.32889 .19946 Mdot
.18592 .23331 Mdot
.83178 .15396 Mdot
.30246 .24267 Mdot
.2166 .19985 Mdot
.26077 .20036 Mdot
.85666 .15019 Mdot
.61978 .14974 Mdot
.68163 .11445 Mdot
.17678 .19937 Mdot
.38274 .20584 Mdot
.36433 .22243 Mdot
.62776 .13587 Mdot
.7937 .15383 Mdot
.73551 .10713 Mdot
.1543 .2006 Mdot
.71799 .15325 Mdot
.37183 .20089 Mdot
.38232 .20927 Mdot
.66268 .1201 Mdot
.7627 .15329 Mdot
.16953 .22699 Mdot
.67512 .15398 Mdot
.85477 .1397 Mdot
.80937 .11841 Mdot
.14136 .20821 Mdot
.33988 .19935 Mdot
.62427 .15163 Mdot
.27343 .20032 Mdot
.28109 .24556 Mdot
.20539 .23911 Mdot
.65315 .15411 Mdot
.85924 .14769 Mdot
.80525 .15403 Mdot
.35031 .22879 Mdot
.18728 .19937 Mdot
.22883 .20007 Mdot
.84065 .15349 Mdot
.62087 .14106 Mdot
.14523 .20271 Mdot
.74362 .15314 Mdot
.35984 .19973 Mdot
.76838 .10913 Mdot
.69206 .11202 Mdot
.17168 .19946 Mdot
.38176 .20445 Mdot
.65846 .15414 Mdot
.32302 .19955 Mdot
.19538 .23634 Mdot
.14729 .21491 Mdot
.6992 .15353 Mdot
.83429 .1279 Mdot
.25426 .20034 Mdot
.85438 .15117 Mdot
.61838 .14848 Mdot
.6389 .15365 Mdot
.29877 .20001 Mdot
.23697 .24494 Mdot
.37008 .21932 Mdot
.63289 .13286 Mdot
.7876 .15371 Mdot
.82683 .15408 Mdot
.31308 .24048 Mdot
.21044 .19973 Mdot
.73193 .15922 Mdot
.41878 .18793 Mdot
.80666 .10904 Mdot
.09702 .18846 Mdot
.13566 .18856 Mdot
.78445 .16022 Mdot
.69801 .10307 Mdot
.32832 .19225 Mdot
.90067 .14753 Mdot
.63033 .16388 Mdot
.67971 .16084 Mdot
.37706 .1892 Mdot
.09293 .20139 Mdot
.26456 .25382 Mdot
.59529 .16556 Mdot
.23451 .19392 Mdot
.88 .16552 Mdot
.91221 .16314 Mdot
.15625 .23219 Mdot
.28721 .19398 Mdot
.59453 .13787 Mdot
.18311 .19149 Mdot
.43754 .19545 Mdot
.37268 .23034 Mdot
.56421 .16275 Mdot
.8348 .16311 Mdot
.46911 .45101 Mdot
.49275 .52664 Mdot
.47414 .45825 Mdot
.49347 .52331 Mdot
.49398 .53162 Mdot
.46411 .43831 Mdot
.46569 .44418 Mdot
.50614 .52493 Mdot
.48713 .47152 Mdot
.49377 .47624 Mdot
.48035 .46531 Mdot
.4961 .52183 Mdot
.50043 .52235 Mdot
.28692 .25254 Mdot
.58719 .16564 Mdot
.87131 .16522 Mdot
.9077 .16445 Mdot
.17827 .23987 Mdot
.27636 .19419 Mdot
.46439 .43383 Mdot
.82473 .16245 Mdot
.49708 .53791 Mdot
.56242 .1604 Mdot
.22373 .19357 Mdot
.39319 .2217 Mdot
.11126 .18785 Mdot
.75369 .15931 Mdot
.39546 .18824 Mdot
.76175 .10078 Mdot
.65311 .11415 Mdot
.15454 .18966 Mdot
.43158 .18911 Mdot
.65018 .1626 Mdot
.34897 .19099 Mdot
.13536 .22366 Mdot
.08594 .19287 Mdot
.70105 .15989 Mdot
.86824 .13113 Mdot
.89585 .16554 Mdot
.2198 .25013 Mdot
.574 .16508 Mdot
.25623 .19428 Mdot
.30874 .19324 Mdot
.91387 .15775 Mdot
.61287 .16489 Mdot
.42325 .2065 Mdot
.56884 .15254 Mdot
.80563 .16128 Mdot
.85429 .16434 Mdot
.32846 .24489 Mdot
.20399 .19269 Mdot
.74281 .1592 Mdot
.42571 .18832 Mdot
.78418 .10395 Mdot
.10373 .18799 Mdot
.14496 .18907 Mdot
.79509 .16072 Mdot
.67553 .10775 Mdot
.33873 .19163 Mdot
.88641 .13975 Mdot
.64015 .16325 Mdot
.69033 .16032 Mdot
.38642 .18867 Mdot
.08722 .19634 Mdot
.88822 .16564 Mdot
.24222 .25301 Mdot
.24535 .19416 Mdot
.29801 .19366 Mdot
.9146 .16102 Mdot
.60388 .1653 Mdot
.57927 .1459 Mdot
.19348 .19212 Mdot
.43291 .20012 Mdot
.35084 .23826 Mdot
.56831 .16422 Mdot
.84467 .16374 Mdot
.46603 .43145 Mdot
.50673 .5515 Mdot
.46423 .4344 Mdot
.50098 .54398 Mdot
.51321 .55873 Mdot
.47429 .43223 Mdot
.46946 .43064 Mdot
.49273 .52603 Mdot
.46609 .44517 Mdot
.46977 .45209 Mdot
.46422 .43912 Mdot
.4965 .53688 Mdot
.49367 .53077 Mdot
.53631 .56762 Mdot
.50375 .464 Mdot
.53433 .57036 Mdot
.4991 .45683 Mdot
.50676 .47024 Mdot
.53485 .55715 Mdot
.53652 .56308 Mdot
.4802 .43621 Mdot
.52577 .56906 Mdot
.51977 .56484 Mdot
.53074 .57092 Mdot
.49324 .4493 Mdot
.48672 .44215 Mdot
.53505 .55763 Mdot
.5079 .47483 Mdot
.53657 .56347 Mdot
.5066 .46981 Mdot
.50724 .47822 Mdot
.5267 .5436 Mdot
.53169 .55083 Mdot
.49276 .44873 Mdot
.53411 .57049 Mdot
.5304 .57086 Mdot
.53622 .5679 Mdot
.50345 .46348 Mdot
.4987 .45626 Mdot
.50122 .5226 Mdot
.48136 .46633 Mdot
.48816 .47235 Mdot
.47502 .45935 Mdot
.49375 .52296 Mdot
.49665 .52177 Mdot
.52052 .53652 Mdot
.50039 .47931 Mdot
.50467 .47977 Mdot
.49472 .4768 Mdot
.50709 .52549 Mdot
.51375 .53027 Mdot
.53106 .57096 Mdot
.49372 .44987 Mdot
.52619 .56928 Mdot
.48723 .44266 Mdot
.4995 .45739 Mdot
.53639 .56734 Mdot
.53454 .57023 Mdot
.46978 .43067 Mdot
.51371 .55925 Mdot
.50721 .55207 Mdot
.52025 .56524 Mdot
.48068 .4366 Mdot
.4747 .43245 Mdot
.52583 .5425 Mdot
.50413 .47983 Mdot
.50697 .47858 Mdot
.49961 .47907 Mdot
.51271 .52944 Mdot
.51951 .5355 Mdot
.53646 .56267 Mdot
.50692 .47066 Mdot
.50404 .46451 Mdot
.50792 .47545 Mdot
.53103 .54975 Mdot
.53465 .55665 Mdot
.49424 .47652 Mdot
.50662 .52521 Mdot
.5 .4792 Mdot
.51323 .52985 Mdot
.50082 .52247 Mdot
.48085 .46582 Mdot
.48765 .47194 Mdot
.53136 .5503 Mdot
.50711 .47841 Mdot
.50792 .47515 Mdot
.5044 .47981 Mdot
.52002 .53601 Mdot
.52627 .54305 Mdot
.46416 .43871 Mdot
.49678 .53739 Mdot
.49382 .53119 Mdot
.50138 .54454 Mdot
.46624 .4313 Mdot
.46431 .43411 Mdot
.47458 .4588 Mdot
.4936 .52313 Mdot
.49637 .52179 Mdot
.49273 .52633 Mdot
.46589 .44468 Mdot
.46944 .45155 Mdot
.12067 .18797 Mdot
.76616 .15957 Mdot
.40538 .18792 Mdot
.73606 .09967 Mdot
.62793 .12317 Mdot
.16586 .19038 Mdot
.43646 .19074 Mdot
.66194 .16186 Mdot
.36053 .19024 Mdot
.11421 .21372 Mdot
.08844 .19035 Mdot
.71347 .15951 Mdot
.84437 .12129 Mdot
.90365 .16504 Mdot
.19396 .24444 Mdot
.58187 .16556 Mdot
.26874 .19427 Mdot
.32096 .19265 Mdot
.90772 .15209 Mdot
.62361 .16429 Mdot
.40611 .21563 Mdot
.56309 .15806 Mdot
.81756 .162 Mdot
.865 .16493 Mdot
.30261 .25042 Mdot
.21622 .19327 Mdot
.75532 .15934 Mdot
.43235 .18927 Mdot
.7584 .10048 Mdot
.11245 .18785 Mdot
.156 .18975 Mdot
.8072 .16138 Mdot
.64977 .11524 Mdot
.3505 .19089 Mdot
.86527 .12982 Mdot
.6517 .1625 Mdot
.70267 .15983 Mdot
.39679 .18819 Mdot
.08606 .19247 Mdot
.89694 .1655 Mdot
.21644 .24953 Mdot
.25786 .19429 Mdot
.31034 .19317 Mdot
.91343 .15714 Mdot
.61424 .16482 Mdot
.56771 .15339 Mdot
.20557 .19277 Mdot
.42136 .2076 Mdot
.32509 .24575 Mdot
.57495 .16517 Mdot
.85571 .16442 Mdot
.69127 .10428 Mdot
.13842 .18871 Mdot
.73519 .1592 Mdot
.42095 .18801 Mdot
.43666 .19667 Mdot
.5895 .14039 Mdot
.78765 .16036 Mdot
.09068 .19969 Mdot
.68288 .16068 Mdot
.8969 .14533 Mdot
.79992 .10733 Mdot
.09892 .18827 Mdot
.3799 .18903 Mdot
.59781 .1655 Mdot
.29045 .19389 Mdot
.25787 .2538 Mdot
.14982 .22969 Mdot
.64313 .16306 Mdot
.9132 .16261 Mdot
.83778 .1633 Mdot
.36624 .23282 Mdot
.1862 .19168 Mdot
.23776 .194 Mdot
.88252 .16558 Mdot
.56523 .16327 Mdot
.12935 .18827 Mdot
.77694 .15992 Mdot
.41348 .18785 Mdot
.7137 .10095 Mdot
.60743 .13181 Mdot
.17594 .19104 Mdot
.43823 .19309 Mdot
.67234 .16124 Mdot
.37034 .18961 Mdot
.09996 .20595 Mdot
.09295 .18903 Mdot
.72431 .1593 Mdot
.82235 .11361 Mdot
.27961 .19414 Mdot
.90924 .16413 Mdot
.58955 .16564 Mdot
.63325 .1637 Mdot
.33146 .19207 Mdot
.1716 .23769 Mdot
.38725 .22432 Mdot
.56265 .16122 Mdot
.82777 .16265 Mdot
.87396 .16533 Mdot
.28021 .25315 Mdot
.22695 .19369 Mdot
.71705 .10063 Mdot
.12803 .18822 Mdot
.72269 .15932 Mdot
.4123 .18784 Mdot
.43815 .19267 Mdot
.61037 .13049 Mdot
.77533 .15986 Mdot
.10179 .20703 Mdot
.67078 .16133 Mdot
.90889 .15295 Mdot
.82569 .11468 Mdot
.09218 .18919 Mdot
.36888 .1897 Mdot
.27799 .19417 Mdot
.90849 .1643 Mdot
.28356 .25287 Mdot
.58837 .16565 Mdot
.63179 .16379 Mdot
.32989 .19216 Mdot
.17492 .2388 Mdot
.82625 .16255 Mdot
.39024 .22301 Mdot
.17442 .19094 Mdot
.22534 .19363 Mdot
.87265 .16527 Mdot
.56251 .16082 Mdot
.1194 .18794 Mdot
.76454 .15953 Mdot
.40412 .18796 Mdot
.7394 .09965 Mdot
.63115 .12192 Mdot
.16437 .19029 Mdot
.43596 .19048 Mdot
.6604 .16195 Mdot
.35903 .19033 Mdot
.11672 .21498 Mdot
.08793 .19061 Mdot
.71184 .15955 Mdot
.8476 .12252 Mdot
.26711 .19428 Mdot
.90271 .16513 Mdot
.58079 .16552 Mdot
.62219 .16438 Mdot
.31937 .19273 Mdot
.19732 .24532 Mdot
.40865 .21437 Mdot
.5635 .15746 Mdot
.81602 .1619 Mdot
.86363 .16486 Mdot
.30598 .24984 Mdot
.21462 .1932 Mdot
.42666 .18841 Mdot
.67215 .10861 Mdot
.10481 .18795 Mdot
.74444 .15921 Mdot
.79668 .1608 Mdot
.4318 .20096 Mdot
.14638 .18915 Mdot
.8839 .13849 Mdot
.08676 .19573 Mdot
.34028 .19154 Mdot
.3878 .1886 Mdot
.78081 .10335 Mdot
.69194 .16025 Mdot
.23886 .25271 Mdot
.6052 .16525 Mdot
.88941 .16565 Mdot
.91472 .16061 Mdot
.13238 .22234 Mdot
.29962 .1936 Mdot
.19504 .19221 Mdot
.84613 .16384 Mdot
.57738 .147 Mdot
.56908 .16438 Mdot
.24698 .19418 Mdot
.3475 .23935 Mdot
.69464 .10365 Mdot
.13704 .18863 Mdot
.73356 .15921 Mdot
.41988 .18797 Mdot
.43715 .19604 Mdot
.59198 .13914 Mdot
.78605 .16029 Mdot
.09174 .20052 Mdot
.6813 .16076 Mdot
.89883 .14644 Mdot
.80329 .10816 Mdot
.09796 .18836 Mdot
.37848 .18911 Mdot
.59655 .16554 Mdot
.28883 .19394 Mdot
.26121 .25384 Mdot
.15302 .23095 Mdot
.64164 .16315 Mdot
.91273 .16288 Mdot
.8363 .1632 Mdot
.36947 .23159 Mdot
.18465 .19159 Mdot
.23614 .19396 Mdot
.88127 .16555 Mdot
.56469 .16302 Mdot
.28997 .23235 Mdot
.67989 .13324 Mdot
.22824 .21128 Mdot
.78446 .14107 Mdot
.79707 .13712 Mdot
.25491 .23467 Mdot
.25383 .21049 Mdot
.72514 .11922 Mdot
.20922 .21376 Mdot
.32128 .21738 Mdot
.31467 .22476 Mdot
.69459 .1252 Mdot
.76195 .1427 Mdot
.7256 .14294 Mdot
.31933 .21546 Mdot
.77192 .12225 Mdot
.20457 .22157 Mdot
.20518 .21525 Mdot
.75274 .14292 Mdot
.7378 .11866 Mdot
.28088 .21066 Mdot
.79374 .13017 Mdot
.68351 .13904 Mdot
.70077 .14193 Mdot
.30453 .212 Mdot
.22253 .22967 Mdot
.68726 .1281 Mdot
.23679 .21086 Mdot
.77064 .14232 Mdot
.30782 .22768 Mdot
.27817 .23402 Mdot
.67922 .13564 Mdot
.79036 .14001 Mdot
.20321 .21679 Mdot
.74477 .14302 Mdot
.74846 .11896 Mdot
.71312 .12071 Mdot
.21481 .21262 Mdot
.3211 .2197 Mdot
.70894 .14244 Mdot
.31106 .21291 Mdot
.21413 .22686 Mdot
.20275 .219 Mdot
.73515 .14303 Mdot
.76072 .1202 Mdot
.79799 .13499 Mdot
.24249 .23363 Mdot
.26344 .21045 Mdot
.28986 .21096 Mdot
.78805 .12726 Mdot
.68836 .14034 Mdot
.79799 .13537 Mdot
.24446 .23386 Mdot
.67918 .13528 Mdot
.26188 .21045 Mdot
.28844 .2109 Mdot
.78913 .12774 Mdot
.68746 .14016 Mdot
.30909 .22721 Mdot
.68829 .12762 Mdot
.76928 .14239 Mdot
.7895 .1402 Mdot
.28015 .23381 Mdot
.23535 .21092 Mdot
.20845 .21397 Mdot
.76049 .14274 Mdot
.3211 .21704 Mdot
.72716 .11907 Mdot
.69598 .12474 Mdot
.22692 .21137 Mdot
.31556 .22429 Mdot
.70756 .14237 Mdot
.31009 .21274 Mdot
.21535 .22733 Mdot
.20289 .2194 Mdot
.73359 .14303 Mdot
.76262 .12049 Mdot
.26766 .23473 Mdot
.68035 .13736 Mdot
.2444 .21064 Mdot
.79415 .13885 Mdot
.79704 .13292 Mdot
.23279 .23209 Mdot
.27148 .2105 Mdot
.70397 .12254 Mdot
.2205 .2119 Mdot
.31918 .22191 Mdot
.30044 .22999 Mdot
.68289 .13052 Mdot
.77735 .14185 Mdot
.20299 .21712 Mdot
.74321 .14303 Mdot
.75049 .1191 Mdot
.715 .12042 Mdot
.21381 .21278 Mdot
.32129 .2193 Mdot
.2969 .21134 Mdot
.78153 .12485 Mdot
.69352 .14118 Mdot
.71633 .14273 Mdot
.31551 .21392 Mdot
.20878 .22441 Mdot
.74632 .14301 Mdot
.32086 .22011 Mdot
.74641 .11885 Mdot
.20348 .21647 Mdot
.21585 .21246 Mdot
.77199 .14224 Mdot
.71127 .12103 Mdot
.29948 .21153 Mdot
.77861 .12396 Mdot
.69576 .14145 Mdot
.71929 .14281 Mdot
.31693 .21438 Mdot
.20715 .22347 Mdot
.24055 .23337 Mdot
.68929 .14052 Mdot
.26501 .21045 Mdot
.79793 .13461 Mdot
.7869 .12679 Mdot
.21297 .22638 Mdot
.29126 .21103 Mdot
.68628 .12857 Mdot
.23823 .21081 Mdot
.3065 .22814 Mdot
.27616 .23421 Mdot
.67933 .136 Mdot
.79118 .1398 Mdot
.32018 .21604 Mdot
.73366 .11874 Mdot
.20372 .2207 Mdot
.72868 .14299 Mdot
.75577 .14286 Mdot
.31798 .22282 Mdot
.2063 .21472 Mdot
.79215 .12924 Mdot
.21958 .22878 Mdot
.28385 .21074 Mdot
.30678 .21226 Mdot
.76843 .12151 Mdot
.70333 .14212 Mdot
.26352 .23482 Mdot
.68117 .13794 Mdot
.7953 .13833 Mdot
.79623 .13206 Mdot
.22928 .23137 Mdot
.27456 .21054 Mdot
.22291 .21167 Mdot
.77978 .14163 Mdot
.70072 .12336 Mdot
.68165 .13144 Mdot
.24743 .21057 Mdot
.29722 .23082 Mdot
.22169 .21179 Mdot
.77858 .14174 Mdot
.31861 .22236 Mdot
.70232 .12295 Mdot
.68224 .13098 Mdot
.24591 .2106 Mdot
.29885 .23041 Mdot
.72713 .14297 Mdot
.31978 .21574 Mdot
.20412 .22113 Mdot
.20572 .21498 Mdot
.75426 .1429 Mdot
.73572 .11869 Mdot
.2982 .21143 Mdot
.7801 .1244 Mdot
.69463 .14132 Mdot
.71781 .14278 Mdot
.31624 .21414 Mdot
.20793 .22393 Mdot
.26559 .23479 Mdot
.68074 .13765 Mdot
.79475 .1386 Mdot
.79667 .1325 Mdot
.23102 .23174 Mdot
.27303 .21052 Mdot
.7634 .14265 Mdot
.31372 .22524 Mdot
.72313 .11941 Mdot
.21003 .21355 Mdot
.22958 .2112 Mdot
.78551 .14092 Mdot
.69325 .12566 Mdot
.312 .21309 Mdot
.75878 .11994 Mdot
.71034 .14251 Mdot
.73671 .14304 Mdot
.3214 .21773 Mdot
.20268 .21862 Mdot
.79298 .12971 Mdot
.22103 .22923 Mdot
.28237 .2107 Mdot
.30567 .21212 Mdot
.7702 .12187 Mdot
.70204 .14203 Mdot
.67962 .13366 Mdot
.25539 .21047 Mdot
.28813 .23268 Mdot
.25285 .23457 Mdot
.68419 .13927 Mdot
.79736 .1368 Mdot
.55366 .50273 Mdot
.57553 .58532 Mdot
.54569 .49171 Mdot
.57214 .59249 Mdot
.57768 .57468 Mdot
.5679 .52801 Mdot
.56091 .51426 Mdot
.55635 .60319 Mdot
.52489 .46684 Mdot
.51026 .45073 Mdot
.5364 .48015 Mdot
.56791 .59766 Mdot
.56277 .60126 Mdot
.46439 .43996 Mdot
.49595 .53588 Mdot
.46654 .44618 Mdot
.4934 .52995 Mdot
.50021 .54286 Mdot
.46565 .43176 Mdot
.46411 .43501 Mdot
.49724 .52176 Mdot
.47592 .46044 Mdot
.48238 .46732 Mdot
.47048 .45319 Mdot
.49276 .52545 Mdot
.49407 .52265 Mdot
.44505 .49597 Mdot
.42754 .41076 Mdot
.43792 .48408 Mdot
.42441 .41862 Mdot
.43157 .40514 Mdot
.46166 .51833 Mdot
.45278 .50702 Mdot
.56607 .52403 Mdot
.57699 .55864 Mdot
.5773 .57797 Mdot
.43084 .46905 Mdot
.42278 .43088 Mdot
.57417 .54552 Mdot
.52126 .58312 Mdot
.475 .41429 Mdot
.46072 .40287 Mdot
.49272 .43204 Mdot
.54796 .60242 Mdot
.53662 .59654 Mdot
.4725 .53107 Mdot
.44247 .39873 Mdot
.43644 .40114 Mdot
.45027 .39862 Mdot
.50318 .56457 Mdot
.48626 .54638 Mdot
.47012 .45264 Mdot
.49274 .52573 Mdot
.47547 .4599 Mdot
.4939 .5228 Mdot
.49353 .53035 Mdot
.4643 .43953 Mdot
.46631 .44567 Mdot
.50758 .52579 Mdot
.48868 .47274 Mdot
.49519 .47706 Mdot
.48187 .46682 Mdot
.49694 .52176 Mdot
.50163 .52275 Mdot
.47388 .43203 Mdot
.51928 .56444 Mdot
.51271 .55821 Mdot
.52535 .56882 Mdot
.48622 .44164 Mdot
.47972 .43583 Mdot
.46417 .4347 Mdot
.50059 .54341 Mdot
.49622 .53638 Mdot
.50626 .55093 Mdot
.46915 .43062 Mdot
.46583 .4316 Mdot
.53006 .57079 Mdot
.49228 .44817 Mdot
.52492 .56856 Mdot
.48572 .44115 Mdot
.49828 .45569 Mdot
.53612 .56817 Mdot
.53388 .5706 Mdot
.46884 .43061 Mdot
.51221 .55769 Mdot
.5058 .55036 Mdot
.51879 .56402 Mdot
.47925 .43546 Mdot
.47348 .43183 Mdot
.52713 .54415 Mdot
.50493 .47972 Mdot
.50736 .47803 Mdot
.50076 .47942 Mdot
.51427 .53071 Mdot
.52102 .53703 Mdot
.53661 .56387 Mdot
.50642 .46937 Mdot
.50314 .46296 Mdot
.50787 .4745 Mdot
.532 .55137 Mdot
.53523 .55812 Mdot
.29558 .21126 Mdot
.78291 .12531 Mdot
.23461 .23243 Mdot
.69245 .14104 Mdot
.71487 .14268 Mdot
.31473 .2137 Mdot
.20969 .22488 Mdot
.79351 .13909 Mdot
.26972 .23465 Mdot
.2429 .21067 Mdot
.26993 .21048 Mdot
.79736 .13334 Mdot
.68002 .13705 Mdot
.44595 .3983 Mdot
.49404 .55482 Mdot
.43916 .39976 Mdot
.47869 .53804 Mdot
.51195 .57375 Mdot
.46704 .40735 Mdot
.45489 .39993 Mdot
.20282 .21747 Mdot
.74165 .14304 Mdot
.75252 .11927 Mdot
.71691 .12014 Mdot
.21284 .21295 Mdot
.32141 .21891 Mdot
.79444 .13064 Mdot
.22408 .23009 Mdot
.68287 .13879 Mdot
.27938 .21063 Mdot
.30336 .21187 Mdot
.7736 .12264 Mdot
.69952 .14182 Mdot
.29177 .23201 Mdot
.68021 .13282 Mdot
.78339 .14122 Mdot
.79673 .13743 Mdot
.25697 .23475 Mdot
.25228 .2105 Mdot
.70566 .12216 Mdot
.21933 .21203 Mdot
.31968 .22147 Mdot
.30198 .22956 Mdot
.68361 .13006 Mdot
.7761 .14195 Mdot
.72407 .14292 Mdot
.31883 .21518 Mdot
.20509 .22202 Mdot
.20469 .21552 Mdot
.75121 .14295 Mdot
.73987 .11867 Mdot
.45768 .40117 Mdot
.51666 .57856 Mdot
.44803 .39834 Mdot
.49853 .55963 Mdot
.53303 .5938 Mdot
.48796 .4271 Mdot
.47087 .41055 Mdot
.4592 .5153 Mdot
.43508 .40204 Mdot
.43044 .40643 Mdot
.44076 .39918 Mdot
.48235 .5421 Mdot
.46943 .52755 Mdot
.54343 .4888 Mdot
.57111 .594 Mdot
.56667 .59874 Mdot
.57474 .58743 Mdot
.55907 .51114 Mdot
.55168 .49985 Mdot
.50592 .44605 Mdot
.55439 .60332 Mdot
.54534 .60151 Mdot
.56125 .60195 Mdot
.53366 .4769 Mdot
.5214 .46295 Mdot
.56537 .59972 Mdot
.54108 .48584 Mdot
.55963 .60251 Mdot
.53075 .47352 Mdot
.54964 .497 Mdot
.57388 .58935 Mdot
.57004 .59539 Mdot
.48326 .42232 Mdot
.54251 .60021 Mdot
.52915 .59051 Mdot
.55228 .60324 Mdot
.51771 .45886 Mdot
.5014 .44121 Mdot
.43611 .48065 Mdot
.42379 .42121 Mdot
.42292 .436 Mdot
.42663 .41255 Mdot
.45069 .50416 Mdot
.44315 .49301 Mdot
.57678 .58083 Mdot
.56426 .5204 Mdot
.55721 .50813 Mdot
.57154 .53717 Mdot
.42876 .46342 Mdot
.57779 .56656 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[35]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo000fool5
000Tool4000Wool2000Xool3000Uool2000Wool3000Hool003Soo`03003ooooo02Coo`03003ooooo
02Ooo`04003ooooo000Xool00`00ooooo`0Sool01000ooooo`009_oo00@00?ooool001Ooo`00>?oo
00<00?ooool09Ooo00<00?ooool0:?oo00<00?ooool09?oo1@009ooo00<00?ooool09?oo00@00?oo
ool001Ooo`00>?oo00<00?ooool09_oo00<00?ooool0:?oo00<00?ooool08ooo00@00?ooool002Ko
o`8002Ooo`<001Soo`00>?oo00<00?ooool09?oo00@00?ooool002Koo`04003ooooo000Vool00`00
ool0000Vool00`00ooooo`0Wool00`00ooooo`0Gool003Ooo`8002Ooo`8002Soo`8002Soo`8002Ko
o`<002Ooo`<001Ooo`00oooo8Ooo000?ool00`00ooooo`3oool?ool000ooo`03003ooooo0?ooo`oo
o`003ooo00<00?ooool0oooo3ooo000?ool00`00ooooo`3oool?ool000Soool001<000Koo`003ooo
00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01_oo00<00?ooool01Ooo
00<00?ooool01Ooo00<00?ooool01_oo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo
00<00?ooool01_oo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01_oo
00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01_oo00<00?ooool01Ooo00<00?ooool01Ooo
00<00?ooool01Ooo00<00?ooool01_oo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo
00<00?ooool01_oo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01_oo00<00?ooool01Ooo
00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01_oo000?ool00`00ooooo`0Vool00`00oooo
o`0Vool00`00ooooo`0Wool00`00ooooo`0Vool00`00ooooo`0Dool00`00ooooo`0?ool00`00oooo
o`0Wool00`00ooooo`0Fool000ooo`03003ooooo0;Woo`03003ooooo05;oo`003ooo00<00?ooool0
oooo3ooo000?ool00`00ooooo`3Cool00`00ooooo`0hool000ooo`800:Soo`03003ooooo02Woo`03
003ooooo00;oo`03003ooooo03?oo`003ooo00<00?ooool0Yooo00<00?ooool0;_oo00<00?ooool0
<ooo000?ool00`00ooooo`2Sool00`00ooooo`1Xool000ooo`03003ooooo0:?oo`03003ooooo03Wo
o`03003ooooo02coo`003ooo00<00?ooool0WOoo00<00?ooool08Ooo0P007?oo00@00?ooool002_o
o`003ooo0P00Vooo00@00?ooool002?oo`8001ooo`03003ooooo02Woo`003ooo00<00?ooool0V?oo
00<00?oo0000Booo00<00?ooool09Ooo000?ool00`00ooooo`22ool4000Bool00`00ooooo`1;ool0
0`00ool0000Uool000ooo`03003ooooo07ooo`03003oo`0000@000coo`03003oo`0005?oo`04003o
oooo000Rool000ooo`03003ooooo07coo`07003oo`00003oo`00ool00`003?oo00<00?oo0000E_oo
00@00?ooool001ooo`003ooo00<00?ooool0Nooo0P0000?oo`00ool00ooo0P002Ooo00<00?ooool0
C_oo00<00?ooool03?oo00<00?oo00007Ooo000?ool2001hool01@00ool00?oo00001Ooo00<00?oo
00001_oo00D00?oo003oo`0000goo`03003ooooo043oo`03003ooooo00koo`05003oooooool0000I
ool000ooo`03003oo`0000<007?oo`800003ool00?oo00Goo`8000Soo`03003oo`0000ooo`03003o
oooo05Goo`8001Soo`003ooo00<00?oo00001@00Kooo00<00?oo00001_oo00@00?ooool000Coo`03
003oo`0006goo`03003oo`0001Koo`003ooo00<00?ooool00P0000Coo`00003oo`8006Woo`04003o
oooo0007ool20007ool00`00ool0001_ool01000ooooo`004ooo000?ool01P00oooooooo003oo`80
0005ool00000ool0001Uool00`00ool00007ool01000ooooo`000ooo00<00?oo0000Mooo00<00?oo
00004Ooo000?ool20004ool500001?oo00000000H_oo0P001ooo00<00?oo00001Ooo0P0000?oo`00
ool0<_oo00h00?oo003oo`00ool00?oo003oo`00ool00?oo0P0000Coo`00ool002Koo`03003ooooo
00Woo`04003ooooo000>ool000ooo`03003oo`0000Goo`800003ool0000000D005coo`04003ooooo
0005ool01000ooooo`001?oo00@00?ooool000goo`03003ooooo01ooo`05003oo`00ool00002ool5
00000ooo0000000300000ooo0000000200001_oo003oo`00ool000;oo`03003ooooo01ooo`03003o
oooo00coo`03003ooooo00coo`003ooo00<00?oo00000_oo00<00?ooool00ooo0P0000?oo`000000
1000Eooo0P0000?oo`00ool01Ooo00<00?oo00001Ooo00<00?oo00004?oo00<00?ooool07Ooo00@0
0?oo003oo`@00003ool00000008000;oo`@000?oo`<000?oo`<000;oo`05003oo`00ool0000kool0
00ooo`03003ooooo00;oo`03003oo`0000Koo`800003ool0000000<005?oo`04003oo`000004ool0
1@00oooooooo00001?oo00@00?ooool002goo`04003oo`00ool30002ool50002ool01000ooooo`00
0_oo00@00?ooool000;oo`04003ooooo0002ool20003ool400000ooo003oo`0Sool00`00ooooo`0B
ool000Koo`D000Coo`03003ooooo00Coo`03003ooooo00Goo`03003oo`0000D004ooo`800003ool0
0?oo00Goo`8000Coo`05003oo`00ool0000<ool00`00ooooo`0Nool200000ooo003oo`050003ool0
1000ooooo`000_oo00@00?ooool000;oo`04003ooooo0002ool01000ooooo`000_oo00D00?ooool0
0?oo00@00007ool00000ool00?oo00007ooo00<00?ooool04_oo0008ool00`00ooooo`04ool00`00
ooooo`08ool00`00ool00005ool4001<ool200000ooo00000003ool20003ool01000ooooo`000_oo
00<00?oo00003_oo00<00?ooool07?oo00<00?oo00000`000_oo00@00?oo003oo`800005ool00?oo
ool0000Dool01@00ooooo`00ool00P000ooo100000Coo`00ool003;oo`002?oo00<00?ooool01?oo
0P002Ooo00<00?oo00000_oo00<00?ooool0C_oo10000_oo00@00?ooooooo`8000Goo`8000_oo`03
003ooooo02;oo`04003oo`00ool400001_oo003ooooo003oo`8000Ooo`l000Soo`800003ool00?oo
00;oo`8000;oo`8000;oo`03003ooooo02goo`002?oo00<00?ooool01?oo00<00?ooool03Ooo00<0
0?oo0000C?oo1P0000?oo`0000001Ooo00<00?oo00000_oo00<00?ooool02_oo00<00?ooool07ooo
00<00?oo00000P000_oo00H00?ooool00?oo0009oolG0007ool01P00ooooo`00ool000;oo`D00003
ool00?oo02coo`002?oo00<00?ooool01?oo00<00?ooool01?oo00<00?ooool02?oo00D00?oooooo
o`0004Soo`<00005ool00000ool00004ool01@00ool00?oo00002Ooo00<00?ooool08_oo00<00?oo
ool01@000ooo00<00?oo00002?oo20003_oo1`001ooo00D00?oo003oo`0000;oo`8000;oo`04003o
oooo000Zool000Ooo`8000Koo`03003ooooo00Coo`03003ooooo00coo`03003oo`0000;oo`03003o
oooo047oo`<000Goo`8000;oo`03003ooooo00_oo`03003ooooo023oo`04003oo`000002ool00`00
ooooo`02ool00`00ool00008ool5000Gool50007ool01@00ool00?oo00001?oo10000_oo00<00?oo
ool09Ooo000?ool00`00ooooo`0:ool00`00ooooo`08ool00`00ooooo`020015ool00`00ool00002
ool20009ool00`00ooooo`0Tool00`00ooooo`040005ool00`00ool0000:ool3000Kool3000;ool3
0003ool20003ool01000ool000009?oo000?ool00`00ooooo`0:ool00`00ooooo`0<ool00`00oooo
o`02ool2000jool20002ool200000ooo003oo`0<ool00`00ooooo`0Qool00`00ooooo`0300000ooo
003oo`04ool3000<ool2000Mool2000<ool30006ool500000ooo003oo`0Qool000ooo`80017oo`03
003ooooo00_oo`8000;oo`80033oo`04003oo`000002ool20002ool00`00ooooo`09ool00`00oooo
o`0Sool01000ooooo`000_oo0`001_oo0`003Ooo3@001ooo3@003_oo0P001Ooo0`000ooo0P008Ooo
000?ool00`00ooooo`0@ool00`00ooooo`02ool00`00ooooo`0:ool20003ool01000ool00?oo0P00
8?oo0P0000Ooo`00ool00?oo003oo`02000<ool00`00ooooo`03ool00`00ooooo`0Sool01`00oooo
oooo003oo`0000Ooo`80013ooal0013oo`8000Koo`03003oo`0000<00003ool0000001koo`003ooo
00<00?ooool05Ooo00<00?ooool00ooo00<00?ooool02Ooo00@00?oo003oo`8000?oo`800004ool0
0?oo0002ool200001?oo003oo`000_oo0P0000Koo`00003oo`00ool200000ooo003oo`0200001?oo
003oo`002ooo0P001?oo00<00?ooool09_oo00<00?ooool00_oo0`002Ooo0P007?oo1`007?oo0P00
2?oo100000Goo`00003oo`0001coo`003ooo00<00?ooool06ooo00<00?ooool00_oo00<00?ooool0
3Ooo0P0000Coo`00ool000;oo`800004ool00?oo0002ool200001_oo00000?oo003oo`800003ool0
0?oo00coo`03003ooooo00;oo`8002goo`03003ooooo00@000[oo`P000Woo`03003oo`0000<00003
ool0000000800003ool00?oo00@00003ool00?oo00@00003ool00?oo00800004ool000000009ool7
000:ool30003ool2000Kool000ooo`03003ooooo023oo`03003ooooo00?oo`03003ooooo02_oo`03
003ooooo00;oo`03003ooooo02goo`03003ooooo00?oo`<000goo`P00004ool00000ool200001?oo
00000?oo100000?oo`0000000P0000?oo`00ool0100000?oo`00ool0100000?oo`00ool00P0000?o
o`0000000P0000Coo`00003oo`<00003ool0000000@000goo`<000;oo`05003oooooool0000Gool0
00ooo`8002Ooo`03003ooooo00;oo`03003ooooo00;oo`03003ooooo01coo`03003ooooo00;oo`03
003ooooo03;oo`03003ooooo00?oo`8001Goo`05003oo`00003oo`0900001?oo00000?oo0P0000?o
o`0000000P0000Coo`00003oo`800004ool00000ool900001?oo00000?oo0`005Ooo0P001_oo00<0
0?ooool05Ooo000?ool00`00ooooo`0[ool00`00ooooo`02ool00`00ooooo`02ool00`00ooooo`02
ool00`00ooooo`02ool00`00ooooo`02ool00`00ooooo`02ool00`00ooooo`03ool00`00ooooo`0m
ool3000;ool900001?oo00000?oo<@0000Coo`00003oo`<000[oo`@001koo`003ooo00<00?ooool0
=Ooo00<00?ooool00_oo00<00?ooool00_oo00<00?ooool00_oo00<00?ooool00_oo00<00?ooool0
@ooo3@0000?oo`000000?`0000?oo`0000003`003_oo00<00?ooool03Ooo000?ool00`00ooooo`1l
ool00`00ool0000CoolJ000VoolK000Aool00`00ooooo`0=ool000ooo`P0077oo`L00003ool00?oo
00_ooaD003kooaD001_oo`003ooo4`00Fooo4@002?oo4`00D_oo4`004_oo000?ool20006oolE0017
oolE0008ool00`00ool0000200000ooo00000005001Xool=000>ool000ooo`80017ooaD0037ooaD0
01?oo`03003oo`00008007Woo`@000koo`003ooo0P003ooo9P004_oo7@0000?oo`0000001`003_oo
00<00?ooool0?_oo00<00?ooool00_oo00<00?ooool0B_oo000?ool00`00ooooo`0=oolB00000ooo
0000001100000ooo000000080009ool01@00oooooooo0000=_oo00<00?ooool00_oo00<00?ooool0
0_oo00<00?ooool00_oo00<00?ooool00_oo00<00?ooool00_oo00<00?ooool00ooo00<00?ooool0
>_oo000?ool00`00ooooo`05ool20006ool00`00ooooo`0Eool:00001?oo00000?oo4`0000Coo`00
003oo`800003ool0000000D00003ool0000001Goo`8000Woo`03003ooooo02koo`03003ooooo00;o
o`03003ooooo00;oo`03003ooooo00;oo`03003ooooo00coo`03003ooooo00;oo`03003ooooo00?o
o`03003ooooo00;oo`03003ooooo03Goo`003ooo00<00?ooool01Ooo0P001_oo0`003ooo1P0000Co
o`00003oo`800006ool00000ool00?oo0P0000?oo`00ool0100000?oo`000000100000?oo`00ool0
0P0000?oo`00ool0100000Coo`00003oo`800005ool00000003oo`05000>ool30004ool00`00oooo
o`0]ool00`00ooooo`03ool00`00ooooo`02ool00`00ooooo`0Vool00`00ooooo`02ool00`00oooo
o`0`ool000ooo`8000[oo`8000?oo`<000_oo`T00004ool00000ool200001_oo00000?oo003oo`80
0003ool00?oo00@00003ool00?oo00@00003ool00?oo00800003ool00?oo00@00004ool00000ool2
00001Ooo00000000ool01`003?oo0P001Ooo00<00?ooool0:?oo00<00?ooool00_oo00<00?ooool0
4ooo0P0000Coo`00000000;oo`800006ool00000003oo`004ooo00<00?ooool00_oo00<00?ooool0
:ooo000?ool00`00ooooo`09ool20002ool00`00ooooo`02000:ool3000nool20009ool200001Ooo
003ooooo0000:?oo00<00?ooool00ooo00<00?ooool03?oo0P0000Goo`00ool00?oo00<000;oo`80
0004ool000000002ool200002Ooo00000000ool00?oo003oo`030002ool2000>ool00`00ooooo`03
ool00`00ooooo`0Uool000ooo`03003ooooo00goo`03003ooooo00800003ool0000000Ooo`8001;o
oa/001;oo`8000Soo`<00005ool00?ooool0000Xool00`00ooooo`0=ool200000ooo003oo`020000
1Ooo003oo`00ool00`004ooo00<00?oo00000P000_oo0P0000?oo`00ool00P000_oo00<00?ooool0
2_oo00<00?ooool09Ooo0006ool40005ool00`00ooooo`0@ool00`00ooooo`030006ool3000>oolQ
000>ool30005ool200000ooo00000002ool00`00ooooo`0Tool00`00ooooo`0<ool200001?oo003o
oooo0P0000?oo`00ool09ooo00@00?oo000000;oo`80017oo`03003ooooo01ooo`001_oo00<00?oo
ool01_oo00<00?ooool04?oo00H00?ooool00?oo0002ool00`00ooooo`03ool2000=ool4000Iool4
000<ool40005ool30002ool00`00ooooo`0Wool00`00ooooo`08ool20002ool200000ooo003oo`0d
ool01`00oooooooo003oo`0000_oo`03003ooooo00;oo`03003ooooo01[oo`001ooo00<00?ooool0
1Ooo0P005?oo00D00?ooool00?oo00<000Coo`800003ool00?oo00Woo`8001goo`8000_oo`@000?o
o`800004ool00000000Vool00`00ooooo`09ool00`00ool00002ool2000oool00`00ool00002ool2
000<ool00`00ooooo`0Jool000Soo`03003ooooo00Coo`03003ooooo01Koo`03003oo`00008000Go
o`05003oo`00ool00008ool4000Iool40009ool00`00ool00003ool500001?oo003oo`008?oo00<0
0?ooool00ooo00<00?ooool02?oo0P0000?oo`00ool0Aooo0P001?oo00<00?ooool03?oo00<00?oo
ool04ooo0006ool01000ooooo`001Ooo00<00?ooool06?oo00@00?oo000000;oo`09003oooooool0
0?oo003oo`0000Ooo`H001?oo`H000Soo`03003oo`0000?oo`03003oo`0000<000;oo`03003ooooo
023oo`03003ooooo00Woo`03003oo`0000;oo`04003ooooo0002ool40018ool00`00ool0000<ool0
0`00ooooo`0Cool000Ooo`8000Koo`03003ooooo01_oo`03003ooooo00<000Goo`03003oo`0000Oo
o`/000Goo`X000Woo`03003oo`0000?oo`<00005ool00?ooool0000Nool00`00ooooo`0=ool02000
ooooo`00ool00?oo0002ool20002ool4001:ool00`00ooooo`02000>ool00`00ooooo`0=ool000oo
o`03003ooooo01koo`<000;oo`8000?oo`04003ooooo0008oolC0007ool00`00ool00002ool00`00
ool00003ool300000ooo003oo`0Pool00`00ooooo`0:ool200000ooo003oo`04ool20002ool00`00
ooooo`0200000ooo003oo`1=ool2000>ool00`00ooooo`0=ool000ooo`03003ooooo023oo`03003o
o`0000@000Coo`05003oo`00003oo`020009ool5000;ool01`00ooooo`00ool00?oo00D00003ool0
0000008002goo`03003ooooo008000;oo`8000?oo`04003oooooool200000ooo003oo`1;ool20009
ool00`00ool0000Gool000ooo`8000?oo`03003ooooo01goo`03003ooooo00<00005ool00000003o
o`0200001Ooo00000?oo00000_oo00<00?ooool02Ooo00@00?ooool000;oo`04003ooooo0003ool0
0`00ooooo`0300000ooo003oo`02000Mool00`00ooooo`0=ool00`00ool00006ool20004ool00`00
ooooo`02001>ool400000ooo003oo`04ool00`00ool00002ool00`00ooooo`0Bool000ooo`03003o
oooo00;oo`03003ooooo023oo`04003oo`00ool30003ool20002ool01000ooooo`000_oo00@00?oo
ool000;oo`04003ooooo0002ool01000ooooo`000ooo100000?oo`00ool00P0000?oo`00ool07ooo
00<00?ooool02_oo00@00?ooool000;oo`03003oo`0000Ooo`03003oo`0005;oo`@00003ool00000
008000Ooo`8001?oo`003ooo00<00?ooool0:Ooo00D00?ooool00?oo00@000?oo`<00005ool00?oo
ool00002ool01000oooooooo0P0000?oo`0000000P000_oo100000?oo`00ool0;_oo0P0000?oo`00
ool01?oo00<00?oo00001ooo00<00?ooool0E?oo2P001Ooo00<00?ooool04Ooo000?ool00`00oooo
o`0;ool00`00ooooo`0Nool01000ool00?oo2@000_oo100000?oo`0000000`0000?oo`0000000`00
0_oo00<00?oo00007_oo00<00?ooool03?oo00@00?ooooooo`8000;oo`03003oo`0000Soo`03003o
oooo05[oo`800003ool0000000D000Soo`03003ooooo00coo`003ooo00<00?ooool02ooo00<00?oo
ool08_oo1P0000Goo`0000000?oo00@0000<ool00000003oo`00ool00?oo003oo`008ooo00<00?oo
ool02ooo0P001ooo00<00?oo00001_oo00<00?oo0000Gooo100000Coo`00000000Ooo`03003ooooo
00coo`003ooo0P0000Coo`00ool003Coo`07003oo`00ool00?oo00000_oo00<00?oo0000=_oo00@0
0?ooool000Goo`03003ooooo00Ooo`03003oo`0006?oo`@00003ool00?oo008001?oo`003ooo00H0
0?ooool00?oo001dool00`00ool00006ool00`00ool00005ool01@00ool00?oo0000Iooo2@004Ooo
000?ool00`00ooooo`07ool00`00ooooo`1[ool00`00ool00006ool00`00ool00005ool01@00ool0
0?oo0000Jooo20004?oo000?ool00`00ooooo`07ool00`00ool0000@ool00`00ooooo`1Eool01000
ooooo`001ooo0P001Ooo00<00?oo0000L_oo1@004?oo000?ool00`00ooooo`09ool00`00ooooo`0>
ool00`00ooooo`10ool00`00ooooo`0@ool00`00ool00007ool01000ooooo`001?oo00<00?oo0000
Mooo0P004?oo000?ool00`00ooooo`0<ool00`00ool0001>ool00`00ooooo`0=ool01000ooooo`00
1ooo00<00?oo00001?oo00@00?oo000008_oo`003ooo0P003Ooo00<00?oo0000G?oo00<00?oo0000
2Ooo0P001?oo0`0000?oo`00ool0Rooo000?ool00`00ooooo`1Xool01000ooooo`002ooo2000Sooo
000?ool00`00ooooo`0Dool00`00ooooo`1=ool01@00oooooooo00003_oo1P00TOoo000?ool00`00
ooooo`0Dool00`00ool0001;ool00`00ool0002Yool000ooo`03003ooooo01Koo`03003ooooo00;o
o`03003ooooo04Coo`03003ooooo0:Woo`003ooo00<00?ooool06ooo00<00?ooool08_oo00<00?oo
ool06_oo00<00?ooool0[_oo000?ool2000Oool00`00ooooo`0Oool00`00ooooo`0Jool00`00oooo
o`2^ool000ooo`03003ooooo01koo`03003ooooo03Woo`03003ooooo0;7oo`003ooo00<00?ooool0
EOoo00<00?ooool00_oo00<00?ooool0/Ooo0007ool20006ool00`00ooooo`1Eool00`00ooooo`2f
ool000Koo`04003ooooo0005ool00`00ooooo`0Zool00`00ooooo`3Qool000Soo`03003ooooo00Co
o`8002_oo`03003ooooo0>7oo`002Ooo00<00?ooool00ooo00<00?ooool0Aooo00<00?ooool0a?oo
0006ool01000ooooo`001Ooo00<00?ooool0Aooo00<00?ooool0a?oo0007ool20006ool00`00oooo
o`3oool?ool000ooo`03003ooooo0?ooo`ooo`003ooo00<00?ooool0oooo3ooo000?ool2003oool@
ool000ooo`03003ooooo0?ooo`ooo`003ooo00<00?ooool0oooo3ooo000?ool00`00ooooo`3oool?
ool000ooo`03003ooooo0?ooo`ooo`003ooo00<00?ooool0oooo3ooo000?ool2003oool@ool000oo
o`03003ooooo0?ooo`ooo`003ooo00<00?ooool0oooo3ooo000?ool00`00ooooo`3oool?ool000oo
o`03003ooooo0?ooo`ooo`003ooo0P00oooo4?oo000?ool00`00ooooo`3oool?ool000ooo`03003o
oooo0?ooo`ooo`003ooo00<00?ooool0K_oo0`00WOoo000?ool00`00ooooo`1/ool500000ooo0000
002Jool000ooo`03003ooooo00Soo`<0063oo`<000Coo`8009[oo`003ooo0P001ooo1`00G_oo00<0
0?ooool01ooo0P00Uooo000?ool00`00ooooo`05ool8001Mool00`00ooooo`08ool3002Fool000oo
o`03003ooooo00Coo`<000Goo`8005coo`03003ooooo00[oo`03003ooooo09Coo`002?oo0`001?oo
00<00?ooool00_oo0`002?oo00<00?ooool0FOoo00<00?ooool0X_oo0009ool00`00ooooo`03ool0
1000oooooooo0`002Ooo0P00F_oo00<00?ooool03Ooo00<00?ooool0T_oo0006ool50004ool20002
ool00`00ooooo`0:ool00`00ooooo`1Xool00`00ool0002Bool000Koo`04003ooooo0005ool01000
ooooo`003?oo0P00F_oo00<00?ooool02Ooo0`000ooo0P00TOoo0007ool00`00ool00005ool4000<
ool2001Jool00`00ooooo`08ool50003ool00`00ooooo`2?ool000Soo`8000Goo`<000;oo`<000Wo
o`03003ooooo05Soo`03003ooooo00Soo`04003oooooool2002Dool000ooo`8000;oo`@000Woo`03
003ooooo06?oo`03003ooooo00;oo`8000?oo`03003ooooo08goo`003ooo00<00?ooool01@002Ooo
00<00?ooool0I?oo00<00?ooool00_oo00L00?ooooooo`00ool0001nool00`00ooooo`0<ool000oo
o`D00003ool0000000Soo`03003ooooo06Goo`03003ooooo00Coo`04003oooooool2001lool2000>
ool000ooo`<000;oo`8000Woo`03003ooooo06Goo`8000Goo`03003ooooo00;oo`03003ooooo07[o
o`8000koo`003ooo0`0000Coo`000000077oo`<000Goo`8007koo`8000ooo`003ooo0P0000Coo`00
000007Coo`03003ooooo00?oo`8000Coo`03003ooooo07Ooo`03003ooooo00koo`003ooo00<00?oo
00000P00J?oo00<00?ooool02_oo00<00?oo00001Ooo00@00?ooooooo`8007Koo`03003ooooo013o
o`003ooo10002ooo00<00?ooool0Fooo0P003Ooo00<00?ooool00ooo00<00?ooool00_oo0P00M?oo
0P004_oo000?ool2000<ool2001Nool00`00ooooo`0;ool30004ool00`00ooooo`02ool00`00oooo
o`1aool3000Bool000ooo`03003ooooo00_oo`03003ooooo06coo`<000?oo`03003ooooo00?oo`03
003ooooo073oo`03003ooooo01;oo`003ooo00<00?ooool02_oo0P00L?oo1@001Ooo0`00Kooo00<0
0?ooool04ooo000?ool00`00ooooo`0:ool2001Pool00`00ooooo`0=ool50006ool2001^ool2000E
ool000ooo`03003ooooo00_oo`03003ooooo05koo`8001_oo`03003ooooo06[oo`<001Goo`003ooo
0P002_oo0P00H_oo00<00?ooool06Ooo0P00Jooo00<00?ooool05Ooo000?ool00`00ooooo`09ool2
001Sool00`00ooooo`0Iool00`00ooooo`1Xool00`00ooooo`0Fool000ooo`03003ooooo00Woo`03
003ooooo06;oo`8001[oo`8006Soo`8001Soo`003ooo00<00?ooool02Ooo00<00?ooool0Hooo00<0
0?ooool06Ooo0P00Iooo0P006?oo000?ool00`00ooooo`07ool2001Wool00`00ooooo`0Iool2001V
ool00`00ooooo`0Gool000ooo`03003ooooo00Ooo`8006Ooo`8001_oo`03003ooooo06?oo`03003o
oooo01Soo`003ooo0P001ooo0P00JOoo00<00?ooool06Ooo0P00Hooo0P006_oo000?ool00`00oooo
o`06ool00`00ooooo`1Zool00`00ooooo`0Hool2001Rool2000Jool000ooo`03003ooooo00Goo`03
003ooooo06_oo`03003ooooo01Woo`03003ooooo063oo`03003ooooo01Woo`003ooo00<00?ooool0
1Ooo00<00?ooool0Jooo00<00?ooool01_oo0`004?oo0P00H?oo00<00?ooool06_oo0007ool20006
ool00`00ooooo`03ool00`00ool0001`ool00`00ooooo`03ool300000ooo003oo`0>ool00`00oooo
o`1Nool00`00ooooo`0Jool000Koo`04003ooooo0005ool00`00ooooo`03ool00`00ooooo`1`ool0
0`00ooooo`03ool01000oooooooo0`003Ooo00<00?ooool0G_oo00<00?ooool06_oo0009ool00`00
ooooo`03ool20003ool00`00ooooo`1aool00`00ooooo`03ool00`00ooooo`02ool2000=ool00`00
ooooo`1Nool00`00ooooo`0:ool2000>ool000Ooo`8000Koo`04003oooooool2001eool00`00oooo
o`02ool00`00ooooo`03ool2000<ool00`00ooooo`1Xool4000>ool000Ooo`03003ooooo00Goo`03
003ooooo008007Koo`8000?oo`03003ooooo00?oo`<000_oo`03003ooooo06Ooo`D000koo`001ooo
0`001Ooo00@00?oo000007Soo`03003ooooo00;oo`03003ooooo00Coo`03003ooooo00[oo`03003o
oooo06Goo`<00003ool0000000koo`003ooo0`00N_oo00D00?ooooooo`0000Koo`8000_oo`03003o
oooo05_oo`03003ooooo00Ooo`800004ool00000000>ool000ooo`8007_oo`03003ooooo00;oo`80
00Goo`8006Soo`03003ooooo00Koo`8000;oo`8000ooo`003ooo00<00?ooool0O?oo00<00?ooool0
0P001Ooo0P00Iooo00<00?ooool01_oo0P000_oo0P004?oo000?ool00`00ooooo`1lool20003ool2
0004ool2000:ool00`00ooooo`1Hool20008ool50003ool00`00ooooo`0;ool000ooo`03003ooooo
07goo`04003oo`00ool30003ool2000:ool00`00ooooo`1Hool00`00ooooo`07ool40003ool2000=
ool000ooo`03003ooooo07ooo`05003oooooool00004ool00`00ooooo`08ool00`00ooooo`1Hool0
0`00ooooo`07ool30003ool2000>ool000ooo`03003ooooo08?oo`03003oo`0000<000[oo`03003o
oooo05Soo`8000goo`8000ooo`003ooo0P00P_oo00@00?ooooooo`<000_oo`03003ooooo05Soo`80
00_oo`03003oo`00013oo`003ooo00<00?ooool0POoo0P004?oo00<00?ooool0F?oo00<00?ooool0
2_oo00<00?ooool04?oo000?ool00`00ooooo`22ool00`00ooooo`0>ool00`00ooooo`1Hool2000:
ool2000Bool000ooo`03003ooooo08Coo`03003ooooo00_oo`8005[oo`8000Woo`8001?oo`003ooo
00<00?ooool0Q?oo00<00?ooool02ooo00<00?ooool0F_oo0P001_oo0`005?oo000?ool20027ool2
000:ool00`00ooooo`1Jool400000ooo003oo`03000Eool000ooo`03003ooooo08Koo`<000Soo`80
05goo`D00003ool00?oo01Koo`003ooo00<00?ooool0R?oo0P001Ooo0`00Gooo10006Ooo000?ool0
0`00ooooo`29ool8001mool000ooo`03003ooooo08[oo`H007koo`003ooo00<00?ooool0oooo3ooo
000?ool2003oool@ool000ooo`03003ooooo0?ooo`ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.364984, -0.441107, \
0.0241755, 0.0355492}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Bouncing Ball", "Subsection"],

Cell[TextData[{
  "This example is a generalization of an example in [",
  ButtonBox["SGT03",
    ButtonData:>{"References.nb", "SGT03"},
    ButtonStyle->"Hyperlink"],
  "].  It models a ball bounding down a ramp with a given profile.  The \
example is good for demonstrating how you can use multiple invocation of ",
  StyleBox["NDSolve", "MR"],
  " with event location to model some behavior."
}], "Text"],

Cell["\<\
This defines a function which computes the solution from one bounce \
to the next.  The solution is computed up until the next time the path \
intersects the ramp.\
\>", "MathCaption"],

Cell[BoxData[
    \(\(OneBounce[k_, \ ramp_]\)[{t0_, x0_, \ xp0_, \ y0_, \ 
          yp0_}]\  := \ \[IndentingNewLine]Module[{sol, \ t1, \ x1, \ xp1, \ 
          y1, \ yp1, \ gramp, \ gp}, \[IndentingNewLine]sol\  = \ 
          First[NDSolve[\[IndentingNewLine]{\(x''\)[t]\  \[Equal] \ 
                  0, \ \(x'\)[t0] \[Equal] \ xp0, \ 
                x[t0]\  \[Equal] \ 
                  x0, \[IndentingNewLine]\(y''\)[
                    t]\  \[Equal] \ \(-9.8\)\ , \ \(y'\)[t0]\  \[Equal] \ 
                  yp0, \ y[t0]\  \[Equal] \ y0}, \[IndentingNewLine]{x, 
                y}, \[IndentingNewLine]{t, t0, \ \[Infinity]}, \ 
              Method \[Rule] {"\<EventLocator\>", \ "\<Event\>" \
\[RuleDelayed] \ y[t]\  - \ ramp[x[t]]}, 
              MaxStepSize \[Rule] 
                0.01]]; \[IndentingNewLine]t1\  = \ \
\(InterpolatingFunctionDomain[
              x\  /. \ sol]\)[\([1, \(-1\)]\)]; \[IndentingNewLine]{x1, xp1, 
            y1, \ yp1}\  = \ \(Reflection[k, \ 
              ramp]\)[{x[t1], \ \(x'\)[t1], \ y[t1], \ \(y'\)[t1]}\  /. \ 
              sol]; \[IndentingNewLine]Sow[{x[t]\  /. sol, 
            t0\  \[LessEqual] \ 
              t\  \[LessEqual] \ \ t1}, \ "\<X\>"]; \[IndentingNewLine]Sow[{\ 
            y[t]\  /. \ sol, 
            t0\  \[LessEqual] \ \ t\  \[LessEqual] \ \ t1}, \ "\<Y\>"]; \
\[IndentingNewLine]Sow[{x1, y1}, \ "\<Bounces\>"]; \[IndentingNewLine]{t1, \ 
          x1, xp1, y1, \ yp1}]\)], "Input",
  CellLabel->"In[36]:="],

Cell["\<\
This defines the function for the bounce when the ball hits the \
ramp. The formula is based on reflection about the normal to the ramp \
assuming only the fraction k of energy is left after a bounce.\
\>", \
"MathCaption"],

Cell[BoxData[
    \(\(Reflection[k_, \ ramp_]\)[{x_, \ xp_, \ y_, \ yp_}]\  := \ 
      Module[{gramp, \ gp, \ xpnew, \ 
          ypnew}, \[IndentingNewLine]gramp\  = \ \(-\(ramp'\)[
              x]\); \[IndentingNewLine]If[
          Not[NumberQ[
              gramp]], \[IndentingNewLine]Print["\<Could not compute \
derivative \>"]; \[IndentingNewLine]Throw[$Failed]]; \[IndentingNewLine]gramp\
\  = \ {\(-\(ramp'\)[x]\), \ 1}; \[IndentingNewLine]If[\ 
          gramp . {xp, yp}\  \[Equal] \ 
            0, \[IndentingNewLine]Print["\<No reflection\>"]; \
\[IndentingNewLine]Throw[$Failed]]; \[IndentingNewLine]gp = \ {1, \(-1\)}\ \
Reverse[gramp]; \[IndentingNewLine]{xpnew, \ 
            ypnew}\  = \ \((k/\((gramp . 
                    gramp)\))\)\ \((gp\ gp . {xp, yp}\  - \ 
                gramp\ gramp . {xp, \ yp})\); \[IndentingNewLine]{x, \ 
          xpnew, \ y, \ ypnew}]\)], "Input",
  CellLabel->"In[37]:="],

Cell["\<\
This defines the function which runs the bouncing ball simulation \
for a given reflection ratio, ramp, and starting position.\
\>", "MathCaption"],

Cell[BoxData[
    \(BouncingBall[k_, \ 
        ramp_, {x0_, y0_}]\  := \ \[IndentingNewLine]Module[{data, \ end, \ 
          bounces, \ xmin, \ xmax, \ ymin, \ ymax}, \[IndentingNewLine]If[
          y0\  < \ ramp[
              x0], \[IndentingNewLine]Print["\<Start above the ramp\>"]; \
\[IndentingNewLine]Return[$Failed]]; \[IndentingNewLine]data\  = 
          Reap[\[IndentingNewLine]Catch[
              Sow[{x0, y0}, \ "\<Bounces\>"]; \[IndentingNewLine]NestWhile[
                OneBounce[k, \ ramp], \ {0, x0, 0, y0, 0}, \ 
                Function[1\  - \ #1[\([1]\)]/#2[\([1]\)]\  > \ 0.01], \ 2, \ 
                25]], _, \ Rule]; \[IndentingNewLine]end\  = \ 
          data[\([1, 1]\)]; \[IndentingNewLine]data\  = \ 
          Last[data]; \[IndentingNewLine]bounces\  = \ \(("\<Bounces\>"\  \
/. \ data)\); \[IndentingNewLine]xmax\  = \ 
          Max[bounces[\([All, 1]\)]]; \[IndentingNewLine]xmin\  = \ 
          Min[bounces[\([All, 1]\)]]; \[IndentingNewLine]ymax\  = \ 
          Max[bounces[\([All, 2]\)]]; \[IndentingNewLine]ymin\  = \ 
          Min[bounces[\([All, 2]\)]]; \[IndentingNewLine]Show[
          Block[{$DisplayFunction\  = \ Identity}, {Plot[
                ramp[x], {x, xmin, \ xmax}, 
                PlotRange \[Rule] {{xmin, \ xmax}, {ymin, \ ymax}}, 
                Epilog \[Rule] {PointSize[ .025], Map[Point, bounces]}, \ 
                AspectRatio \[Rule] \((ymax\  - \ ymin)\)/\((xmax\  - \ 
                        xmin)\)], \ 
              ParametricPlot[
                Evaluate[{Piecewise["\<X\>"\  /. \ data], \ 
                    Piecewise["\<Y\>"\  /. \ data]}], \ {t, 0, end}, \ 
                PlotStyle \[Rule] RGBColor[1, 0, 0]]}]];]\)], "Input",
  CellLabel->"In[38]:="],

Cell[TextData[{
  "This is the example which is done in ",
  " [",
  ButtonBox["SGT03",
    ButtonData:>{"References.nb", "SGT03"},
    ButtonStyle->"Hyperlink"],
  "]",
  "."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(ramp[x_]\  := \ 
        If[x\  < \ 1, 1\  - \ x, \ 0];\)\), "\[IndentingNewLine]", 
    \(BouncingBall[ .7, \ ramp, {0, 1.25}]\)}], "Input",
  CellLabel->"In[39]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .70605 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-2.4503e-17 0.564842 3.66084e-15 0.564842 [
[.14121 -0.0125 -12 -9 ]
[.14121 -0.0125 12 0 ]
[.28242 -0.0125 -9 -9 ]
[.28242 -0.0125 9 0 ]
[.42363 -0.0125 -12 -9 ]
[.42363 -0.0125 12 0 ]
[.56484 -0.0125 -3 -9 ]
[.56484 -0.0125 3 0 ]
[.70605 -0.0125 -12 -9 ]
[.70605 -0.0125 12 0 ]
[.84726 -0.0125 -9 -9 ]
[.84726 -0.0125 9 0 ]
[.98847 -0.0125 -12 -9 ]
[.98847 -0.0125 12 0 ]
[-0.0125 .11297 -18 -4.5 ]
[-0.0125 .11297 0 4.5 ]
[-0.0125 .22594 -18 -4.5 ]
[-0.0125 .22594 0 4.5 ]
[-0.0125 .33891 -18 -4.5 ]
[-0.0125 .33891 0 4.5 ]
[-0.0125 .45187 -18 -4.5 ]
[-0.0125 .45187 0 4.5 ]
[-0.0125 .56484 -6 -4.5 ]
[-0.0125 .56484 0 4.5 ]
[-0.0125 .67781 -18 -4.5 ]
[-0.0125 .67781 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .70605 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.14121 0 m
.14121 .00625 L
s
[(0.25)] .14121 -0.0125 0 1 Mshowa
.28242 0 m
.28242 .00625 L
s
[(0.5)] .28242 -0.0125 0 1 Mshowa
.42363 0 m
.42363 .00625 L
s
[(0.75)] .42363 -0.0125 0 1 Mshowa
.56484 0 m
.56484 .00625 L
s
[(1)] .56484 -0.0125 0 1 Mshowa
.70605 0 m
.70605 .00625 L
s
[(1.25)] .70605 -0.0125 0 1 Mshowa
.84726 0 m
.84726 .00625 L
s
[(1.5)] .84726 -0.0125 0 1 Mshowa
.98847 0 m
.98847 .00625 L
s
[(1.75)] .98847 -0.0125 0 1 Mshowa
.125 Mabswid
.02824 0 m
.02824 .00375 L
s
.05648 0 m
.05648 .00375 L
s
.08473 0 m
.08473 .00375 L
s
.11297 0 m
.11297 .00375 L
s
.16945 0 m
.16945 .00375 L
s
.19769 0 m
.19769 .00375 L
s
.22594 0 m
.22594 .00375 L
s
.25418 0 m
.25418 .00375 L
s
.31066 0 m
.31066 .00375 L
s
.33891 0 m
.33891 .00375 L
s
.36715 0 m
.36715 .00375 L
s
.39539 0 m
.39539 .00375 L
s
.45187 0 m
.45187 .00375 L
s
.48012 0 m
.48012 .00375 L
s
.50836 0 m
.50836 .00375 L
s
.5366 0 m
.5366 .00375 L
s
.59308 0 m
.59308 .00375 L
s
.62133 0 m
.62133 .00375 L
s
.64957 0 m
.64957 .00375 L
s
.67781 0 m
.67781 .00375 L
s
.73429 0 m
.73429 .00375 L
s
.76254 0 m
.76254 .00375 L
s
.79078 0 m
.79078 .00375 L
s
.81902 0 m
.81902 .00375 L
s
.87551 0 m
.87551 .00375 L
s
.90375 0 m
.90375 .00375 L
s
.93199 0 m
.93199 .00375 L
s
.96023 0 m
.96023 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .11297 m
.00625 .11297 L
s
[(0.2)] -0.0125 .11297 1 0 Mshowa
0 .22594 m
.00625 .22594 L
s
[(0.4)] -0.0125 .22594 1 0 Mshowa
0 .33891 m
.00625 .33891 L
s
[(0.6)] -0.0125 .33891 1 0 Mshowa
0 .45187 m
.00625 .45187 L
s
[(0.8)] -0.0125 .45187 1 0 Mshowa
0 .56484 m
.00625 .56484 L
s
[(1)] -0.0125 .56484 1 0 Mshowa
0 .67781 m
.00625 .67781 L
s
[(1.2)] -0.0125 .67781 1 0 Mshowa
.125 Mabswid
0 .02824 m
.00375 .02824 L
s
0 .05648 m
.00375 .05648 L
s
0 .08473 m
.00375 .08473 L
s
0 .14121 m
.00375 .14121 L
s
0 .16945 m
.00375 .16945 L
s
0 .19769 m
.00375 .19769 L
s
0 .25418 m
.00375 .25418 L
s
0 .28242 m
.00375 .28242 L
s
0 .31066 m
.00375 .31066 L
s
0 .36715 m
.00375 .36715 L
s
0 .39539 m
.00375 .39539 L
s
0 .42363 m
.00375 .42363 L
s
0 .48012 m
.00375 .48012 L
s
0 .50836 m
.00375 .50836 L
s
0 .5366 m
.00375 .5366 L
s
0 .59308 m
.00375 .59308 L
s
0 .62133 m
.00375 .62133 L
s
0 .64957 m
.00375 .64957 L
s
.25 Mabswid
0 0 m
0 .70605 L
s
0 0 m
1 0 L
1 .70605 L
0 .70605 L
closepath
clip
newpath
.5 Mabswid
0 .56484 m
.00123 .56362 L
.00245 .56239 L
.00491 .55994 L
.00981 .55503 L
.01963 .54521 L
.04091 .52393 L
.06078 .50406 L
.08026 .48458 L
.10139 .46345 L
.12111 .44373 L
.14248 .42236 L
.16346 .40138 L
.18303 .38181 L
.20426 .36058 L
.22407 .34077 L
.24349 .32135 L
.26457 .30028 L
.28423 .28061 L
.30555 .2593 L
.32647 .23837 L
.34599 .21886 L
.36715 .19769 L
.38691 .17793 L
.40831 .15653 L
.42933 .13551 L
.44894 .1159 L
.4702 .09465 L
.49004 .0748 L
.5095 .05534 L
.53061 .03423 L
.54046 .02438 L
.54539 .01946 L
.55031 .01453 L
.55565 .00919 L
.55832 .00653 L
.56098 .00386 L
.56232 .00252 L
.56365 .00119 L
.56499 0 L
.56632 0 L
.56766 0 L
.56899 0 L
.57166 0 L
.57428 0 L
.5769 0 L
.58214 0 L
.59262 0 L
.61217 0 L
.63337 0 L
Mistroke
.65316 0 L
.67256 0 L
.69362 0 L
.71326 0 L
.73455 0 L
.75443 0 L
.77393 0 L
.79507 0 L
.81481 0 L
.83619 0 L
.85719 0 L
.87677 0 L
.89801 0 L
.91783 0 L
.93727 0 L
.95836 0 L
.97803 0 L
.99936 0 L
1 0 L
Mfstroke
1 0 0 r
0 .70605 m
0 .70604 L
0 .70601 L
0 .70589 L
0 .7054 L
0 .70346 L
0 .6948 L
0 .68121 L
0 .66273 L
0 .63692 L
0 .60741 L
0 .56952 L
0 .56719 L
0 .56484 L
s
0 .56484 m
.00167 .56483 L
.00335 .5648 L
.0067 .56468 L
.01339 .56419 L
.02678 .56225 L
.05581 .55359 L
.08292 .54 L
.1095 .52152 L
.13833 .49571 L
.16523 .4662 L
.19439 .42831 L
.22302 .38514 L
.24972 .33953 L
.26325 .31446 L
.27001 .30143 L
.27339 .29479 L
.27508 .29144 L
.27677 .28807 L
s
.27677 .28807 m
.27912 .28689 L
.28146 .28569 L
.28615 .28322 L
.29552 .27805 L
.31426 .26673 L
.35491 .23775 L
.39286 .20518 L
.43007 .1681 L
.47043 .1221 L
.5081 .07376 L
.52806 .04603 L
.53803 .03161 L
.54302 .02427 L
.54552 .02056 L
.54801 .01683 L
s
.54801 .01683 m
.55099 .01483 L
.55397 .0128 L
.55992 .00865 L
.56588 .00439 L
.56886 .00221 L
.57183 0 L
s
.57183 0 m
.57356 .00128 L
.57528 .00253 L
.57872 .00498 L
.58561 .00964 L
.59939 .01798 L
.62927 .03161 L
.65716 .03884 L
.67084 .04044 L
.67425 .04065 L
.67596 .04072 L
.67767 .04077 L
.67938 .0408 L
.68109 .04081 L
.6828 .0408 L
.68451 .04077 L
.68636 .04072 L
.68822 .04064 L
.69193 .04042 L
.69934 .03969 L
.71418 .0371 L
.74186 .02825 L
.77187 .01276 L
.78121 .00668 L
.78588 .00341 L
.78822 .00173 L
.79056 0 L
s
.79056 0 m
.79176 .00089 L
.79297 .00176 L
.79538 .00344 L
.8002 .00655 L
.80984 .01181 L
.83076 .01875 L
.83564 .01949 L
.83808 .01974 L
.84052 .01991 L
.84174 .01996 L
.84296 .01999 L
.84418 .02 L
.8454 .01999 L
.84662 .01995 L
.84784 .0199 L
.85028 .01973 L
.85268 .01949 L
.85507 .01917 L
.85986 .01828 L
.86943 .01554 L
.89019 .00523 L
.89396 .00271 L
.89584 .00138 L
.89773 0 L
s
.89773 0 m
.89857 .00062 L
.89942 .00122 L
.9011 .00236 L
.90448 .00439 L
.91123 .00749 L
.91855 .00938 L
.92038 .00961 L
.9213 .0097 L
.92221 .00975 L
.92312 .00979 L
.92404 .0098 L
.92495 .00979 L
.92587 .00975 L
.92672 .00969 L
.92758 .00962 L
.92929 .0094 L
.9327 .00872 L
.93954 .00636 L
.94489 .00359 L
.94757 .0019 L
.9489 .00097 L
.95024 0 L
s
.95024 0 m
.95083 .00043 L
.95142 .00084 L
.95261 .0016 L
.95497 .00288 L
.95733 .00383 L
.9597 .00446 L
.96098 .00467 L
.96162 .00474 L
.96226 .00478 L
.9629 .0048 L
.96354 .0048 L
.96418 .00477 L
.96482 .00472 L
.96738 .00427 L
.96994 .00345 L
.97296 .00199 L
.97447 .00106 L
.97522 .00055 L
.97598 0 L
s
.97598 0 m
.97639 .0003 L
.9768 .00058 L
.97763 .00107 L
.97928 .00182 L
.98094 .00225 L
.98135 .0023 L
.98176 .00234 L
.98218 .00235 L
.98259 .00235 L
.98297 .00232 L
.98334 .00229 L
.98409 .00216 L
.98559 .0017 L
.98709 .00099 L
.98783 .00053 L
.98821 .00027 L
.98858 0 L
s
.98858 0 m
.98887 .00021 L
.98916 .00039 L
.98974 .0007 L
.99032 .00093 L
.9909 .00108 L
.99119 .00112 L
.99148 .00115 L
.99177 .00115 L
.99206 .00113 L
.99264 .00104 L
.99322 .00087 L
.99399 .0005 L
.99438 .00027 L
.99476 0 L
s
.99476 0 m
.99495 .00013 L
.99514 .00025 L
.99552 .00042 L
.9959 .00053 L
.99609 .00056 L
.99628 .00056 L
.99647 .00056 L
.99665 .00053 L
.99703 .00042 L
.99741 .00025 L
.9976 .00013 L
.99779 0 L
s
.99779 0 m
.99798 .00012 L
.99816 .00021 L
.99835 .00026 L
.99853 .00028 L
.99872 .00026 L
.9989 .00021 L
.99909 .00012 L
.99927 0 L
s
.99927 0 m
.99936 6e-05 L
.99945 .0001 L
.99955 .00013 L
.99964 .00014 L
.99973 .00013 L
.99982 .0001 L
.99991 6e-05 L
1 0 L
s
0 g
.025 w
0 .70605 Mdot
0 .56484 Mdot
.27677 .28807 Mdot
.54801 .01683 Mdot
.57183 0 Mdot
.79056 0 Mdot
.89773 0 Mdot
.95024 0 Mdot
.97598 0 Mdot
.98858 0 Mdot
.99476 0 Mdot
.99779 0 Mdot
.99927 0 Mdot
1 0 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[39]:=",
  ImageSize->{288, 203.312},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000ba000`40n007h00Ooooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo000_ool20004ool00`00oooo
o`02ool40002ool3000Cool20004ool00`00ooooo`02ool3000Cool20004ool00`00ooooo`03ool0
0`00ooooo`02ool3000Iool4000Gool40003ool00`00ooooo`02ool40002ool3000Bool40003ool0
0`00ooooo`02ool3000Cool40003ool00`00ooooo`03ool00`00ooooo`02ool30003ool002koo`04
003ooooo0008ool00`00ooooo`06ool00`00ooooo`0?ool01000ooooo`002ooo00<00?ooool03ooo
00@00?ooool000Woo`03003ooooo00Goo`03003ooooo01Soo`03003ooooo01Soo`03003ooooo00Oo
o`03003ooooo00Koo`03003ooooo017oo`03003ooooo00[oo`03003ooooo01;oo`03003ooooo00So
o`03003ooooo00Goo`40007oo`7oo`00;_oo00@00?ooool000Woo`03003ooooo00Goo`03003ooooo
00ooo`04003ooooo000;ool00`00ooooo`0?ool01000ooooo`002_oo00<00?ooool01?oo00<00?oo
ool06?oo00<00?ooool06?oo00<00?ooool02?oo00<00?ooool01Ooo00<00?ooool04Ooo00<00?oo
ool02_oo00<00?ooool04_oo00<00?ooool02Ooo00<00?ooool01?oo0@000Ooo0Ooo000^ool01000
ooooo`002_oo00@00?ooooooo`<001;oo`04003ooooo0008ool3000Bool01000ooooo`002_oo00@0
0?ooooooo`<001_oo`03003ooooo01Soo`03003ooooo00Woo`04003oooooool3000Dool00`00oooo
o`07ool3000Eool00`00ooooo`09ool01000oooooooo0`000ooo000^ool01000ooooo`002ooo00@0
0?ooool001Coo`04003ooooo0008ool00`00ooooo`0Bool01000ooooo`002ooo00@00?ooool001go
o`03003ooooo01Soo`03003ooooo00[oo`04003ooooo000Fool00`00ooooo`07ool00`00ooooo`0E
ool00`00ooooo`0:ool01000ooooo`001Ooo000^ool01000ooooo`002?oo00@00?ooool000;oo`03
003ooooo01;oo`04003ooooo0008ool00`00ooooo`0Bool01000ooooo`002?oo00@00?ooool000;o
o`03003ooooo01Woo`<001Soo`<000Woo`04003ooooo0002ool00`00ooooo`0Bool30009ool00`00
ooooo`0Cool30009ool01000ooooo`000_oo00<00?ooool00ooo000_ool2000:ool20003ool4000B
ool20009ool4000Bool20009ool40002ool4000Jool00`00ooooo`0Hool00`00ooooo`08ool20003
ool4000Cool00`00ooooo`07ool4000Dool00`00ooooo`07ool40002ool40002ool00?ooob7oo`00
oooo8Ooo003ooolQool001Goool0008000[oo`005Ooo00<00?ooool01?oo00<00?ooool01?oo00<0
0?ooool01?oo00<00?ooool01Ooo00<00?ooool01?oo00<00?ooool01?oo00<00?ooool01?oo00<0
0?ooool01Ooo00<00?ooool01?oo00<00?ooool01?oo00<00?ooool01?oo00<00?ooool01Ooo00<0
0?ooool01?oo00<00?ooool01?oo00<00?ooool01?oo00<00?ooool01Ooo00<00?ooool01?oo00<0
0?ooool01?oo00<00?ooool01?oo00<00?ooool00ooo00<00?P0000010000_oo00<00?ooool01?oo
00<00?ooool01?oo00<00?ooool01Ooo00<00?ooool01?oo00<00?ooool01?oo00<00?ooool01?oo
00<00?ooool00ooo1@001?oo00<00?ooool01?oo00<00?ooool01?oo00D00?ooooooooP000D000Go
o`03003h0?P000D000;h00P000[oo`005Ooo00<00?ooool0R?oo1@000ooo00@00?oon03h033oo`;h
00;oo`03003ooooo00;h01;oo`;h00?oo`03003oooP000Kh00Coo`03003ooooo00Coo`03003ooooo
00<000[oo`005Ooo00<00?ooool0R?oo1@001ooo0_P0;?oo0_P02Ooo0_P03_oo0_P0:Ooo000Eool0
0`00ooooo`27ool60009ool2n00Xool2n00=ool2n00:ool2n00[ool001Goo`03003ooooo08Soo`D0
00_oo`;h02Coo`;h017oo`[h02goo`005Ooo00<00?ooool0Qooo1P003Ooo0_P08?oo0_P0B_oo000E
ool20027ool01@00oooh0?oo00004Ooo0_P07?oo0_P0C?oo000Eool00`00ooooo`26ool00`00n03o
o`0Eool2n00Fool4n01>ool001Goo`03003ooooo08Goo`03003h0?oo01Soo`Ch00koo`Ch05;oo`00
5Ooo00<00?ooool0Q?oo00<00?oon0007Ooo3_P0E_oo000Eool00`00ooooo`23ool00`00oooh0022
ool001Goo`03003ooooo08;oo`03003oooP008?oo`005Ooo00<00?ooool0POoo00<00?oon000Q?oo
000Eool20021ool01000ooooooP0Q?oo000Eool00`00ooooo`1oool01000ooooooP0QOoo000Eool0
0`00ooooo`1nool01000ooooooP0Q_oo000Eool00`00ooooo`1mool01000ooooooP0Qooo000Eool0
0`00ooooo`1lool01@00oooooooon000Qooo000Eool00`00ooooo`1kool01@00oooooooon000R?oo
000Eool00`00ooooo`1jool01@00oooooooon000ROoo000Eool00`00ooooo`1hool20003ool00oP0
ooooo`28ool001Goo`8007Soo`03003ooooo00?oo`03n03ooooo08Soo`005Ooo00<00?ooool0M_oo
00<00?ooool00ooo00?h0?ooool0ROoo000Eool00`00ooooo`1eool00`00ooooo`03ool00oP0oooo
o`2:ool001Goo`03003ooooo07Coo`03003ooooo00?oo`03n03ooooo08_oo`005Ooo00<00?ooool0
Looo00<00?ooool00ooo00?h0?ooool0S?oo0002ool20004ool00`00ooooo`02ool40004ool00`00
ooooo`1cool00`00ooooo`02ool00oP0ooooo`2=ool00005ool00?ooool00008ool00`00ooooo`05
ool00`00ooooo`1bool00`00ooooo`03ool00oP0ooooo`2=ool00005ool00?ooool00009ool00`00
ooooo`04ool2001bool00`00ooooo`03ool00oP0ooooo`2>ool00005ool00?ooool0000:ool00`00
ooooo`03ool00`00ooooo`1`ool00`00ooooo`03ool00oP0ooooo`2?ool00005ool00?ooool0000;
ool00`00ooooo`02ool00`00ooooo`1_ool00`00ooooo`03ool00oP0ooooo`2@ool00005ool00?oo
ool00008ool01000ooooo`001?oo00<00?ooool0K_oo00<00?ooool00ooo00?h0?ooool0TOoo0002
ool2000:ool20005ool00`00ooooo`1]ool00`00ooooo`03ool00oP0ooooo`2Bool001Goo`03003o
oooo06coo`03003ooooo00Coo`03n03ooooo09;oo`005Ooo00<00?ooool0Jooo00<00?ooool01?oo
00?h0?ooool0Tooo000Eool2001[ool00`00ooooo`04ool00oP0ooooo`2Dool001Goo`03003ooooo
06Woo`03003ooooo00Coo`03n03ooooo09Goo`005Ooo00<00?ooool0J?oo00<00?ooool01?oo00?h
0?ooool0U_oo000Eool00`00ooooo`1Wool00`00ooooo`04ool00oP0ooooo`2Gool001Goo`03003o
oooo06Koo`03003ooooo00Goo`03n03ooooo09Ooo`005Ooo00<00?ooool0IOoo00<00?ooool01Ooo
00?h0?ooool0V?oo000Eool00`00ooooo`1Sool20007ool00oP0ooooo`2Iool001Goo`8006?oo`03
003ooooo00Koo`03n03ooooo09[oo`005Ooo00<00?ooool0HOoo00<00?ooool01_oo00?h0?ooool0
Vooo000Eool00`00ooooo`1Pool00`00ooooo`06ool00oP0ooooo`2Lool001Goo`03003ooooo05oo
o`03003ooooo00Koo`03n03ooooo09goo`005Ooo00<00?ooool0G_oo00<00?ooool01_oo00?h0?oo
ool0W_oo000Eool00`00ooooo`1Nool00`00ooooo`05ool00oP0ooooo`2Oool001Goo`03003ooooo
05goo`03003ooooo00Goo`03n03ooooo0:3oo`005Ooo00<00?ooool0G?oo00<00?ooool01Ooo00?h
0?ooool0XOoo000Eool2001Lool00`00ooooo`05ool00oP0ooooo`2Rool001Goo`03003ooooo05[o
o`03003ooooo00Goo`03n03ooooo0:?oo`005Ooo00<00?ooool0FOoo00<00?ooool01Ooo00?h0?oo
ool0Y?oo000Eool00`00ooooo`1Hool00`00ooooo`05ool00oP0ooooo`2Uool001Goo`03003ooooo
05Ooo`03003ooooo00Goo`03n03ooooo0:Koo`000_oo0P001?oo00<00?ooool01?oo00<00?ooool0
0ooo00<00?ooool0E_oo00<00?ooool01?oo0_P0ZOoo00001Ooo003ooooo00002_oo00<00?ooool0
0ooo00<00?ooool0EOoo00<00?ooool01?oo00?h0?ooool0ZOoo00001Ooo003ooooo00001ooo1@00
1?oo0P00EOoo00<00?ooool01?oo00?h0?ooool0Z_oo00001Ooo003ooooo00001ooo00@00?ooool0
00Goo`03003ooooo05?oo`03003ooooo00Coo`03n03ooooo0:_oo`0000Goo`00ooooo`0000Soo`03
003oo`0000Goo`03003ooooo05;oo`03003ooooo00Coo`03n03ooooo0:coo`0000Goo`00ooooo`00
00Woo`8000Goo`03003ooooo057oo`03003ooooo00Coo`03n03ooooo0:goo`000_oo0P002ooo00<0
0?ooool00ooo00<00?ooool0D?oo00<00?ooool00ooo0_P0/?oo000Eool00`00ooooo`1>ool20005
ool00oP0ooooo`2`ool001Goo`03003ooooo04goo`03003ooooo00?oo`;h0;?oo`005Ooo0P00COoo
00<00?ooool00_oo0_P0]Ooo000Eool00`00ooooo`1;ool00`00ooooo`02ool00oP0ooooo`2eool0
01Goo`03003ooooo04[oo`04003oooooool2n02hool001Goo`03003ooooo04Woo`05003ooooooooh
002jool001Goo`03003ooooo04Woo`04003oooP0n02kool001Goo`03003ooooo04Soo`03003oooP0
0;goo`005Ooo00<00?ooool0@_oo1P0000?oooP0ool0_Ooo000Eool20013ool50002n02oool001Go
o`03003ooooo047oo`H00<7oo`005Ooo00<00?ooool0@_oo1@00`Ooo000Eool00`00ooooo`12ool5
0031ool001Goo`03003ooooo04;oo`03003h00000<?oo`005Ooo00<00?ooool0@Ooo00<00?P0ool0
a?oo000Eool00`00ooooo`10ool00`00oooh0035ool001Goo`03003ooooo03koo`800003oooh0?oo
0<Goo`005Ooo0P00?_oo00D00?ooooooooP00<Koo`005Ooo00<00?ooool0??oo00D00?ooooooooP0
0<Ooo`005Ooo00<00?ooool0>ooo00<00?ooool00_oo00?h0?ooool0aOoo000Eool00`00ooooo`0j
ool00`00ooooo`02ool00oP0ooooo`36ool001Goo`03003ooooo03Woo`03003ooooo00?oo`03n03o
oooo0<Koo`000_oo0P001?oo00<00?ooool00ooo0P001Ooo00<00?ooool0>?oo00<00?ooool00ooo
00?h0?ooool0aooo00001Ooo003ooooo00002?oo00@00?ooool000Coo`03003ooooo03Ooo`03003o
oooo00Coo`03n03ooooo0<Ooo`0000Goo`00ooooo`0000Soo`04003ooooo0004ool2000gool00`00
ooooo`04ool00oP0ooooo`38ool00005ool00?ooool00008ool30005ool00`00ooooo`0eool00`00
ooooo`04ool00oP0ooooo`39ool00005ool00?ooool00008ool00`00ooooo`05ool00`00ooooo`0d
ool00`00ooooo`05ool00oP0ooooo`39ool00005ool00?ooool00008ool00`00ooooo`05ool00`00
ooooo`0cool00`00ooooo`05ool00oP0ooooo`3:ool000;oo`8000[oo`<000Coo`03003ooooo03;o
o`03003ooooo00Koo`03n03ooooo0<[oo`005Ooo00<00?ooool0<Ooo00<00?ooool01_oo00?h0?oo
ool0booo000Eool00`00ooooo`0`ool00`00ooooo`07ool00oP0ooooo`3;ool001Goo`80033oo`03
003ooooo00Ooo`03n03ooooo0<coo`005Ooo00<00?ooool0;ooo00<00?ooool01_oo00?h0?ooool0
cOoo000Eool00`00ooooo`0^ool00`00ooooo`07ool00oP0ooooo`3=ool001Goo`03003ooooo02go
o`03003ooooo00Ooo`03n03ooooo0<koo`005Ooo00<00?ooool0;?oo00<00?ooool02?oo00?h0?oo
ool0c_oo000Eool00`00ooooo`0[ool00`00ooooo`08ool00oP0ooooo`3?ool001Goo`03003ooooo
02[oo`03003ooooo00Soo`03n03ooooo0=3oo`005Ooo0P00:_oo00<00?ooool02Ooo00?h0?ooool0
d?oo000Eool00`00ooooo`0Xool00`00ooooo`09ool00oP0ooooo`3Aool001Goo`03003ooooo02Oo
o`03003ooooo00Woo`03n03ooooo0=;oo`005Ooo00<00?ooool09_oo00<00?ooool02_oo00?h0?oo
ool0d_oo000Eool00`00ooooo`0Tool2000<ool00oP0ooooo`3Cool001Goo`03003ooooo02?oo`03
003ooooo00coo`03n03ooooo0=?oo`005Ooo00<00?ooool08_oo00<00?ooool03?oo00?h0?ooool0
e?oo000Eool00`00ooooo`0Qool00`00ooooo`0<ool00oP0ooooo`3Eool001Goo`80027oo`03003o
oooo00goo`03n03ooooo0=Goo`005Ooo00<00?ooool07ooo00<00?ooool03Ooo00?h0?ooool0e_oo
000Eool00`00ooooo`0Oool00`00ooooo`0<ool00oP0ooooo`3Gool001Goo`03003ooooo01koo`03
003ooooo00coo`03n03ooooo0=Soo`005Ooo00<00?ooool07Ooo00<00?ooool03Ooo00?h0?ooool0
f?oo0002ool20004ool00`00ooooo`03ool20005ool00`00ooooo`0Lool00`00ooooo`0=ool00oP0
ooooo`3Iool00005ool00?ooool00008ool01000ooooo`001?oo00<00?ooool06ooo00<00?ooool0
3Ooo00?h0?ooool0f_oo00001Ooo003ooooo00002?oo00@00?ooool000Coo`8001_oo`03003ooooo
00goo`03n03ooooo0=_oo`0000Goo`00ooooo`0000Woo`8000Goo`03003ooooo01Woo`03003ooooo
00goo`03n03ooooo0=coo`0000Goo`00ooooo`0000Soo`04003ooooo0004ool00`00ooooo`0Hool0
0`00ooooo`0>ool00oP0ooooo`3Lool00005ool00?ooool00008ool01000ooooo`001?oo00<00?oo
ool05ooo00<00?ooool03_oo00?h0?ooool0gOoo0002ool2000:ool20005ool00`00ooooo`0Fool0
0`00ooooo`0>ool00oP0ooooo`3Nool001Goo`03003ooooo01Coo`80013oo`03n03ooooo0=ooo`00
5Ooo00<00?ooool04ooo00<00?ooool03ooo00?h0?ooool0h?oo000Eool2000Cool00`00ooooo`0?
ool00oP0ooooo`3Qool001Goo`03003ooooo017oo`03003ooooo00ooo`03n03ooooo0>;oo`005Ooo
00<00?ooool04?oo00<00?ooool03ooo00?h0?ooool0hooo000Eool00`00ooooo`0?ool00`00oooo
o`0?ool00oP0ooooo`3Tool001Goo`03003ooooo00koo`03003ooooo00ooo`03n03ooooo0>Goo`00
5Ooo00<00?ooool03Ooo00<00?ooool03ooo00?h0?ooool0i_oo000Eool00`00ooooo`0<ool00`00
ooooo`0?ool00oP0ooooo`3Wool001Goo`8000coo`03003ooooo00ooo`03n03ooooo0>Soo`005Ooo
00<00?ooool02_oo00<00?ooool03ooo00?h0?ooool0jOoo000Eool00`00ooooo`09ool00`00oooo
o`0?ool00oP0ooooo`3Zool001Goo`03003ooooo00Soo`03003ooooo00ooo`03n03ooooo0>_oo`00
5Ooo00<00?ooool01ooo00<00?ooool03ooo00?h0?ooool0k?oo000Eool00`00ooooo`06ool00`00
ooooo`0>ool2n03_ool001Goo`03003ooooo00Goo`03003ooooo00goo`;h0?7oo`005Ooo00<00?oo
ool01Ooo00<00?ooool02ooo0_P0looo000Eool20005ool00`00ooooo`0;ool00oP0ooooo`3cool0
01Goo`03003ooooo00?oo`03003ooooo00[oo`;h0?Koo`005Ooo00<00?ooool00_oo00<00?ooool0
2Ooo0_P0n?oo000Eool00`00ooooo`02000:ool2n03jool001Goo`03003oo`0000[oo`;h0?coo`00
3Ooo10001?oo0`001_oo1?P0o_oo000?ool00`00ooooo`03ool30002ool4n03oool3ool000ooo`03
003ooooo00?oo`<000;h0?ooo`Ooo`003ooo00<00?ooool00ooo0`00oooo2Ooo000?ool00`00oooo
o`03ool3003oool9ool000goo`<000Goo`03003ooooo0?ooo`Woo`003ooo00<00?ooool00ooo00?h
0?ooool0oooo2Ooo000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h
0000ool0oooo2Ooo000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h
0?ooool0oooo2Ooo000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h
0?ooool0oooo2Ooo000Eool00oP0003oo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h
0?ooool0oooo2Ooo000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h
0?ooool0oooo2Ooo000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h
0000ool0oooo2Ooo000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h
0?ooool0oooo2Ooo000Eool00oP0ooooo`3oool9ool00003ool00000008000?oo`03003ooooo00;o
o`@000Coo`03n03ooooo0?ooo`Woo`000ooo00<00?ooool01ooo00<00?ooool01Ooo00?h0?ooool0
oooo2Ooo0003ool00`00ooooo`08ool00`00ooooo`04ool00oP0003oo`3oool9ool000?oo`03003o
oooo00Woo`03003ooooo00?oo`03n03ooooo0?ooo`Woo`000ooo00<00?ooool02_oo00<00?ooool0
0_oo00?h0?ooool0oooo2Ooo00001?oo000000002Ooo00@00?ooool000Coo`03n03ooooo0?ooo`Wo
o`000ooo00<00?ooool02?oo0P001Ooo00?h0?ooool0oooo2Ooo000Eool3003oool9ool001Goo`<0
0?ooo`Woo`005Ooo0`00oooo2Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {202.312, 0}} -> {-0.146258, -0.114619, \
0.00689549, 0.00689549}}]
}, Open  ]],

Cell["The ramp is now defined to be a quarter circle", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(circle[x_]\  := \ 
        If[x\  < \ 1, \ Sqrt[1\  - \ x^2], \ 0];\)\), "\[IndentingNewLine]", 
    \(BouncingBall[ .7, \ circle, { .1, 1.25}]\)}], "Input",
  CellLabel->"In[41]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .76879 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.0615031 0.615031 6.54994e-16 0.615031 [
[.09225 -0.0125 -12 -9 ]
[.09225 -0.0125 12 0 ]
[.24601 -0.0125 -9 -9 ]
[.24601 -0.0125 9 0 ]
[.39977 -0.0125 -12 -9 ]
[.39977 -0.0125 12 0 ]
[.70729 -0.0125 -12 -9 ]
[.70729 -0.0125 12 0 ]
[.86104 -0.0125 -9 -9 ]
[.86104 -0.0125 9 0 ]
[.54103 .12301 -18 -4.5 ]
[.54103 .12301 0 4.5 ]
[.54103 .24601 -18 -4.5 ]
[.54103 .24601 0 4.5 ]
[.54103 .36902 -18 -4.5 ]
[.54103 .36902 0 4.5 ]
[.54103 .49202 -18 -4.5 ]
[.54103 .49202 0 4.5 ]
[.54103 .61503 -6 -4.5 ]
[.54103 .61503 0 4.5 ]
[.54103 .73804 -18 -4.5 ]
[.54103 .73804 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .76879 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.09225 0 m
.09225 .00625 L
s
[(0.25)] .09225 -0.0125 0 1 Mshowa
.24601 0 m
.24601 .00625 L
s
[(0.5)] .24601 -0.0125 0 1 Mshowa
.39977 0 m
.39977 .00625 L
s
[(0.75)] .39977 -0.0125 0 1 Mshowa
.70729 0 m
.70729 .00625 L
s
[(1.25)] .70729 -0.0125 0 1 Mshowa
.86104 0 m
.86104 .00625 L
s
[(1.5)] .86104 -0.0125 0 1 Mshowa
.125 Mabswid
.12301 0 m
.12301 .00375 L
s
.15376 0 m
.15376 .00375 L
s
.18451 0 m
.18451 .00375 L
s
.21526 0 m
.21526 .00375 L
s
.27676 0 m
.27676 .00375 L
s
.30752 0 m
.30752 .00375 L
s
.33827 0 m
.33827 .00375 L
s
.36902 0 m
.36902 .00375 L
s
.43052 0 m
.43052 .00375 L
s
.46127 0 m
.46127 .00375 L
s
.49202 0 m
.49202 .00375 L
s
.52278 0 m
.52278 .00375 L
s
.58428 0 m
.58428 .00375 L
s
.61503 0 m
.61503 .00375 L
s
.64578 0 m
.64578 .00375 L
s
.67653 0 m
.67653 .00375 L
s
.73804 0 m
.73804 .00375 L
s
.76879 0 m
.76879 .00375 L
s
.79954 0 m
.79954 .00375 L
s
.83029 0 m
.83029 .00375 L
s
.0615 0 m
.0615 .00375 L
s
.03075 0 m
.03075 .00375 L
s
0 0 m
0 .00375 L
s
.89179 0 m
.89179 .00375 L
s
.92255 0 m
.92255 .00375 L
s
.9533 0 m
.9533 .00375 L
s
.98405 0 m
.98405 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
.55353 .12301 m
.55978 .12301 L
s
[(0.2)] .54103 .12301 1 0 Mshowa
.55353 .24601 m
.55978 .24601 L
s
[(0.4)] .54103 .24601 1 0 Mshowa
.55353 .36902 m
.55978 .36902 L
s
[(0.6)] .54103 .36902 1 0 Mshowa
.55353 .49202 m
.55978 .49202 L
s
[(0.8)] .54103 .49202 1 0 Mshowa
.55353 .61503 m
.55978 .61503 L
s
[(1)] .54103 .61503 1 0 Mshowa
.55353 .73804 m
.55978 .73804 L
s
[(1.2)] .54103 .73804 1 0 Mshowa
.125 Mabswid
.55353 .03075 m
.55728 .03075 L
s
.55353 .0615 m
.55728 .0615 L
s
.55353 .09225 m
.55728 .09225 L
s
.55353 .15376 m
.55728 .15376 L
s
.55353 .18451 m
.55728 .18451 L
s
.55353 .21526 m
.55728 .21526 L
s
.55353 .27676 m
.55728 .27676 L
s
.55353 .30752 m
.55728 .30752 L
s
.55353 .33827 m
.55728 .33827 L
s
.55353 .39977 m
.55728 .39977 L
s
.55353 .43052 m
.55728 .43052 L
s
.55353 .46127 m
.55728 .46127 L
s
.55353 .52278 m
.55728 .52278 L
s
.55353 .55353 m
.55728 .55353 L
s
.55353 .58428 m
.55728 .58428 L
s
.55353 .64578 m
.55728 .64578 L
s
.55353 .67653 m
.55728 .67653 L
s
.55353 .70729 m
.55728 .70729 L
s
.25 Mabswid
.55353 0 m
.55353 .76879 L
s
0 0 m
1 0 L
1 .76879 L
0 .76879 L
closepath
clip
newpath
.5 Mabswid
0 .61195 m
.00123 .61182 L
.00245 .6117 L
.00491 .61143 L
.00981 .61088 L
.01963 .60966 L
.04091 .60644 L
.06078 .60275 L
.08026 .59847 L
.10139 .59307 L
.12111 .58729 L
.14248 .58022 L
.16346 .57241 L
.18303 .56433 L
.20426 .55465 L
.22407 .54471 L
.24349 .53408 L
.26457 .52148 L
.28423 .50866 L
.30555 .4935 L
.32647 .47722 L
.34599 .46067 L
.36715 .44104 L
.38691 .42094 L
.40831 .39691 L
.42933 .3706 L
.44894 .34309 L
.4702 .30913 L
.49004 .27214 L
.5095 .22851 L
.52006 .20013 L
.52533 .18408 L
.53061 .16632 L
.53554 .14767 L
.538 .13733 L
.54046 .1261 L
.54539 .09975 L
.54785 .08339 L
.54908 .07384 L
.55031 .06283 L
.55164 .04809 L
.55298 .02598 L
.55431 0 L
.55565 0 L
.55698 0 L
.55832 0 L
.56098 0 L
.56632 0 L
.57166 0 L
.58214 0 L
Mistroke
.59262 0 L
.61217 0 L
.63337 0 L
.65316 0 L
.67256 0 L
.69362 0 L
.71326 0 L
.73455 0 L
.75443 0 L
.77393 0 L
.79507 0 L
.81481 0 L
.83619 0 L
.85719 0 L
.87677 0 L
.89801 0 L
.91783 0 L
.93727 0 L
.95836 0 L
.97803 0 L
.99936 0 L
1 0 L
Mfstroke
1 0 0 r
0 .76879 m
0 .76875 L
0 .76862 L
0 .7681 L
0 .76604 L
0 .7578 L
0 .72104 L
0 .6634 L
0 .63895 L
0 .62577 L
0 .61894 L
0 .61195 L
s
0 .61195 m
.00072 .61547 L
.00145 .6189 L
.00289 .6255 L
.00578 .63768 L
.01157 .65792 L
.01784 .67366 L
.02097 .6791 L
.02411 .68293 L
.02704 .68505 L
.0285 .68558 L
.02923 .68571 L
.02996 .68576 L
.03069 .68571 L
.03143 .68558 L
.03289 .68506 L
.03582 .68295 L
.04156 .67474 L
.0473 .66111 L
.05975 .61293 L
.06078 .60782 L
.0618 .60254 L
s
.0618 .60254 m
.06331 .60474 L
.06481 .60684 L
.06782 .6108 L
.07384 .61767 L
.07986 .62318 L
.08588 .62731 L
.09241 .63023 L
.09567 .63109 L
.0973 .63137 L
.09893 .63154 L
.10057 .63162 L
.1022 .63159 L
.10383 .63147 L
.10546 .63124 L
.11199 .62932 L
.12417 .62141 L
.13636 .60787 L
.14586 .59341 L
.15061 .5849 L
.15299 .58032 L
.15536 .57553 L
s
.15536 .57553 m
.15759 .57642 L
.15981 .57723 L
.16426 .57859 L
.16871 .57961 L
.17316 .58028 L
.17539 .58048 L
.17762 .58061 L
.17984 .58064 L
.18207 .58059 L
.18429 .58046 L
.18652 .58023 L
.19097 .57953 L
.19579 .57838 L
.20062 .57683 L
.21027 .57251 L
.22957 .55904 L
.24501 .5436 L
.25273 .53433 L
.25659 .52931 L
.25852 .5267 L
.26045 .52403 L
s
.26045 .52403 m
.26311 .52358 L
.26576 .52305 L
.27106 .52173 L
.28166 .51806 L
.30287 .50659 L
.34886 .46284 L
.35498 .45507 L
.35804 .45102 L
.3611 .44685 L
s
.3611 .44685 m
.36376 .44509 L
.36642 .44325 L
.37175 .43932 L
.38241 .43042 L
.40372 .40849 L
.44992 .34208 L
.45195 .33857 L
s
.45195 .33857 m
.4542 .33554 L
.45645 .33242 L
.46095 .32592 L
.46995 .31188 L
.48795 .2797 L
.52698 .19103 L
.56343 .08491 L
.57581 .04376 L
.58199 .0222 L
.58509 .01118 L
.58818 0 L
s
.58818 0 m
.58976 .00569 L
.59133 .0113 L
.59448 .02225 L
.60078 .04313 L
.61339 .08076 L
.64071 .14347 L
.66622 .1787 L
.67873 .18775 L
.68498 .19024 L
.68811 .19098 L
.68967 .19122 L
.69123 .19138 L
.69293 .19145 L
.69462 .19143 L
.69632 .19131 L
.69802 .19108 L
.7048 .1892 L
.71836 .18064 L
.74369 .14764 L
.77113 .08682 L
.79807 .00176 L
.79855 0 L
s
.79855 0 m
.79965 .00397 L
.80076 .00786 L
.80296 .01537 L
.80737 .02936 L
.81619 .05323 L
.83532 .08611 L
.83978 .09006 L
.84425 .09261 L
.84648 .09335 L
.8476 .09359 L
.84871 .09375 L
.84983 .09381 L
.85095 .09379 L
.85206 .09368 L
.85318 .09348 L
.85537 .09283 L
.85756 .09185 L
.86193 .08886 L
.87069 .07883 L
.88968 .03847 L
.89566 .0205 L
.89864 .01056 L
.90014 .00536 L
.90163 0 L
s
.90163 0 m
.9024 .00277 L
.90318 .00545 L
.90472 .01055 L
.90781 .01973 L
.91398 .03397 L
.92068 .04319 L
.92402 .04538 L
.9257 .04587 L
.92653 .04596 L
.92737 .04595 L
.92815 .04585 L
.92893 .04567 L
.9305 .04503 L
.93362 .0427 L
.93987 .03382 L
.95213 5e-05 L
.95214 0 L
s
.95214 0 m
.95268 .00192 L
.95322 .00376 L
.9543 .00718 L
.95647 .01299 L
.95863 .01742 L
.96079 .02048 L
.96196 .02156 L
.96254 .02195 L
.96313 .02224 L
.96372 .02243 L
.9643 .02252 L
.96489 .0225 L
.96547 .02239 L
.96606 .02218 L
.96664 .02186 L
.96782 .02092 L
.97016 .01784 L
.97353 .01059 L
.97521 .00571 L
.97605 .00296 L
s
.97605 .00296 m
.97689 0 L
s
.97689 0 m
.97727 .00133 L
.97765 .00258 L
.97841 .00482 L
.97992 .00827 L
.98143 .01034 L
.98219 .01086 L
.98257 .01099 L
.98294 .01104 L
.98332 .011 L
.9837 .01087 L
.98446 .01036 L
.98598 .00829 L
.9875 .00484 L
.98826 .00259 L
.98864 .00134 L
.98902 0 L
s
.98902 0 m
.98928 .00092 L
.98955 .00176 L
.99008 .00317 L
.99061 .00424 L
.99114 .00496 L
.9914 .0052 L
.99167 .00534 L
.99193 .00541 L
.9922 .00538 L
.99246 .00527 L
.99273 .00508 L
.99326 .00443 L
.99368 .00366 L
.99411 .00266 L
.99454 .00144 L
.99496 0 L
s
.99496 0 m
.99514 .00062 L
.99533 .00116 L
.99569 .00199 L
.99605 .00248 L
.99624 .00261 L
.99642 .00265 L
.9966 .00261 L
.99678 .00248 L
.99715 .00199 L
.99751 .00116 L
.99769 .00062 L
.99787 0 L
s
.99787 0 m
.99805 .00057 L
.99823 .00097 L
.99841 .00122 L
.99859 .0013 L
.99877 .00122 L
.99894 .00097 L
.99912 .00057 L
.9993 0 L
s
.9993 0 m
.99939 .00028 L
.99948 .00048 L
.99956 .0006 L
.99965 .00064 L
.99974 .0006 L
.99983 .00048 L
.99991 .00028 L
1 0 L
s
0 g
.025 w
0 .76879 Mdot
0 .61195 Mdot
.0618 .60254 Mdot
.15536 .57553 Mdot
.26045 .52403 Mdot
.3611 .44685 Mdot
.45195 .33857 Mdot
.58818 0 Mdot
.79855 0 Mdot
.90163 0 Mdot
.95214 0 Mdot
.97689 0 Mdot
.98902 0 Mdot
.99496 0 Mdot
.99787 0 Mdot
.9993 0 Mdot
1 0 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[41]:=",
  ImageSize->{288, 221.375},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000gA000`40n007h00Ooooo8Ooo003ooolQ
ool001Ooo`8000Coo`03003ooooo00;oo`@000;oo`<001Soo`8000Coo`03003ooooo00;oo`<001Wo
o`8000Coo`03003ooooo00?oo`03003ooooo00;oo`<003koo`@000?oo`03003ooooo00;oo`@000;o
o`<001Soo`@000?oo`03003ooooo00;oo`<002Soo`005_oo00@00?ooool000Soo`03003ooooo00Ko
o`03003ooooo01Coo`04003ooooo000;ool00`00ooooo`0Eool01000ooooo`002Ooo00<00?ooool0
1Ooo00<00?ooool0?Ooo00<00?ooool01ooo00<00?ooool01_oo00<00?ooool05ooo00<00?ooool0
2_oo00<00?ooool09Ooo000Fool01000ooooo`002Ooo00<00?ooool01Ooo00<00?ooool05?oo00@0
0?ooool000_oo`03003ooooo01Goo`04003ooooo000:ool00`00ooooo`04ool00`00ooooo`0mool0
0`00ooooo`08ool00`00ooooo`05ool00`00ooooo`0Gool00`00ooooo`0:ool00`00ooooo`0Uool0
01Koo`04003ooooo000:ool01000oooooooo0`005ooo00@00?ooool000Soo`<001Soo`04003ooooo
000:ool01000oooooooo0`00@?oo00<00?ooool02Ooo00@00?ooooooo`<001[oo`03003ooooo00Oo
o`<002Soo`005_oo00@00?ooool000_oo`04003ooooo000Iool01000ooooo`002?oo00<00?ooool0
6?oo00@00?ooool000_oo`04003ooooo0012ool00`00ooooo`0:ool01000ooooo`007?oo00<00?oo
ool01ooo00<00?ooool0:?oo000Fool01000ooooo`002?oo00@00?ooool000;oo`03003ooooo01Oo
o`04003ooooo0008ool00`00ooooo`0Hool01000ooooo`002?oo00@00?ooool000;oo`03003ooooo
03koo`<000Woo`04003ooooo0002ool00`00ooooo`0Hool30009ool00`00ooooo`0Xool001Ooo`80
00[oo`8000?oo`@001Ooo`8000Woo`@001Soo`8000Woo`@000;oo`@003ooo`03003ooooo00Soo`80
00?oo`@001Woo`03003ooooo00Ooo`@002Ooo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo
0008oooo000@0009ool000Soo`03003ooooo00Goo`03003ooooo00Goo`03003ooooo00Koo`03003o
oooo00Goo`03003ooooo00Goo`03003ooooo00Koo`03003ooooo00Goo`03003ooooo00Goo`03003o
oooo00Koo`03003ooooo00Goo`03003ooooo00Goo`03003ooooo00Koo`03003ooooo00Goo`03003o
oooo00Goo`03003ooooo00Koo`03003ooooo00Goo`03003ooooo00Goo`03003ooooo00Goo`8000Ko
o`D000Coo`03003ooooo00Goo`03003ooooo00Koo`03003ooooo00Goo`03003ooooo00Goo`03003o
oooo00Koo`03003ooooo00?oo`D000Goo`03003ooooo00Koo`03003ooooo00Goo`H000;oo`03003o
oooo00?oo`D000;oo`T000Woo`00WOoo0P001ooo00?h0000ool0=_oo00?h0000ool06Ooo00?h0000
ool02_oo00?h0000n0001?oo00Kh0000n03h0000ool20009ool009goo`8000Ooo`03n03oooP003Ko
o`;h01[oo`;h00_oo`03n03oooP000Coo`03n03ooooo00koo`00WOoo00<00?ooool01_oo00?h0?oo
n000=_oo00?h0?oon0006?oo00Ch0?oooooh00[oo`04n03ooooon002ool00oP0ooooo`0?ool009go
o`03003ooooo00Goo`05n03ooooooooh000dool01?P0ooooooP06?oo00Ch0?oooooh00Woo`03n03o
oooo00;oo`?h017oo`00WOoo00<00?ooool01Ooo00Gh0?ooooooooP003Coo`05n03ooooooooh000G
ool01?P0ooooooP02Ooo00?h0?ooool00_oo0_P04_oo002Mool00`00ooooo`05ool01OP0oooooooo
n000=?oo00Gh0?ooooooooP001Koo`03n03ooooo00;oo`03n03ooooo00Koo`03n03ooooo01Koo`00
WOoo0P001_oo00Gh0?ooooooooP003?oo`03n03ooooo00;oo`03n03ooooo01Coo`03n03ooooo00;o
o`03n03ooooo00Goo`03n03ooooo01Ooo`00WOoo00<00?ooool01?oo00?h0?ooool00ooo00?h0?oo
ool0<?oo00?h0?ooool00ooo00?h0?ooool04_oo00?h0?ooool01?oo00?h0?ooool01?oo00?h0?oo
ool05ooo002Mool00`00ooooo`04ool00oP0ooooo`03ool00oP0ooooo`0`ool00oP0ooooo`03ool0
0oP0ooooo`0Bool00oP0ooooo`05ool00oP0ooooo`02ool00oP0ooooo`0Hool009goo`03003ooooo
00Coo`03n03ooooo00?oo`03n03ooooo02ooo`03n03ooooo00Goo`03n03ooooo017oo`03n03ooooo
00Koo`04n03ooooon00Kool009goo`03003ooooo00Coo`03n03ooooo00?oo`03n03ooooo02ooo`03
n03ooooo00Goo`03n03ooooo013oo`03n03ooooo00Soo`;h01coo`00WOoo00<00?ooool00ooo00?h
0?ooool01Ooo00?h0?ooool0;Ooo00?h0?ooool01ooo00?h0?ooool03ooo00?h0?ooool09_oo002M
ool00`00ooooo`03ool00oP0ooooo`05ool00oP0ooooo`0]ool00oP0ooooo`07ool00oP0ooooo`0>
ool00oP0ooooo`0Wool009goo`03003ooooo00?oo`03n03ooooo00Koo`03n03ooooo02coo`03n03o
oooo00Soo`03n03ooooo00goo`03n03ooooo02Ooo`00WOoo0P000ooo00?h0?ooool01ooo00?h0?oo
ool0:ooo00?h0?ooool02Ooo00?h0?ooool03?oo00?h0?ooool0:?oo002Mool00`00ooooo`02ool0
0oP0ooooo`07ool00oP0ooooo`0[ool00oP0ooooo`0:ool00oP0ooooo`0;ool00oP0ooooo`0Xool0
09goo`05003ooooooooh000;ool00oP0ooooo`0Zool00oP0ooooo`0:ool00oP0ooooo`0:ool00oP0
ooooo`0Yool009coo`8000?oo`03n03ooooo00Woo`03n03ooooo02Woo`03n03ooooo00coo`03n03o
oooo00Woo`03n03ooooo02Woo`00W?oo0P000ooo00?h0?ooool02_oo00?h0?ooool0:?oo00?h0?oo
ool03?oo00?h0?ooool02?oo00?h0?ooool0:_oo002Lool20002ool00oP0ooooo`0;ool00oP0oooo
o`0Xool00oP0ooooo`0=ool00oP0ooooo`07ool00oP0ooooo`0Zool009coo`8000;oo`03n03ooooo
00_oo`03n03ooooo02Ooo`03n03ooooo00koo`03n03ooooo00Koo`03n03ooooo02_oo`00W?oo0P00
0_oo00?h0?ooool03?oo00?h0?ooool09_oo00?h0?ooool03ooo00?h0?ooool01Ooo00?h0?ooool0
:ooo002Lool300000oP0ooooo`0=ool00oP0ooooo`0Vool00oP0ooooo`0@ool00oP0ooooo`02ool2
n00^ool009_oo`05003oo`00oooh000@ool00oP0ooooo`0Tool00oP0ooooo`0Bool4n00`ool009_o
o`05003oo`00oooh000@ool00oP0ooooo`0Tool00oP0ooooo`16ool009_oo`04003oo`00n00Aool0
0oP0ooooo`0Sool00oP0ooooo`17ool009_oo`04003oo`00n00Bool00oP0ooooo`0Rool00oP0oooo
o`17ool009_oo`04003oo`00n00Bool00oP0ooooo`0Qool00oP0ooooo`18ool009_oo`03003oooP0
01Coo`03n03ooooo023oo`03n03ooooo04Soo`00R_oo0P001?oo00<00?ooool00_oo100000Goo`00
ooooooP001Coo`03n03ooooo023oo`03n03ooooo04Soo`00ROoo00@00?ooool000Soo`03003ooooo
00;oo`04003ooooon00Eool00oP0ooooo`0Nool00oP0ooooo`19ool008Woo`04003ooooo0009ool0
1`00oooooooo003oooP000<001?oo`03n03ooooo01koo`03n03ooooo04Woo`00ROoo00@00?ooool0
00[oo`07003ooooo003oooP000005Ooo00?h0?ooool07Ooo00?h0?ooool0B_oo0029ool01000oooo
o`002ooo00H00?oo003oooP0000Fool00oP0ooooo`0Lool00oP0ooooo`1:ool008Woo`04003ooooo
0008ool00`00ooooo`0200001?oon03oo`005_oo00?h0?ooool06ooo00?h0?ooool0Booo002:ool2
000:ool200001_oo003oooP0ool001Ooo`03n03ooooo01[oo`03n03ooooo04_oo`00VOoo00D00?P0
ooooo`0001Ooo`03n03ooooo01Woo`03n03ooooo04coo`00VOoo00D00?P0ooooo`0001Soo`03n03o
oooo01Soo`03n03ooooo04coo`00VOoo00D00?P0ooooo`0001Soo`03n03ooooo01Ooo`03n03ooooo
04goo`00V?oo00<00?P0ool00_oo0P006?oo00?h0?ooool05Ooo00?h0?ooool0C_oo002Hool00`00
n03oo`02ool00`00ooooo`0Hool00oP0ooooo`0Dool00oP0ooooo`1>ool009Soo`03003h0?oo00;o
o`03003ooooo01Soo`03n03ooooo01?oo`03n03ooooo04ooo`00Uooo00<00?P0ool00ooo00<00?oo
ool06Ooo00?h0?ooool04Ooo00?h0?ooool0D?oo002Gool00`00n03oo`03ool00`00ooooo`0Jool0
0oP0ooooo`0?ool00oP0ooooo`1Aool009Ooo`03003h0?oo00?oo`03003ooooo01_oo`03n03ooooo
00koo`03n03ooooo057oo`00Uooo00?h0?ooool00ooo00<00?ooool06ooo00?h0?ooool03Ooo00?h
0?ooool0D_oo002Fool00`00n03oo`04ool00`00ooooo`0Lool00oP0ooooo`0;ool00oP0ooooo`1C
ool009Koo`03003h0?oo00Coo`8001koo`;h00Woo`;h05Koo`00U_oo00?h0?ooool01?oo00<00?oo
ool07ooo00?h0?ooool01Ooo00?h0?ooool0E_oo002Eool00`00n03oo`05ool00`00ooooo`0Pool7
n01Iool009Goo`03003h0?oo00Goo`03003ooooo083oo`00U?oo00<00?P0ool01_oo00<00?ooool0
P?oo002Dool00`00n03oo`06ool00`00ooooo`20ool009Coo`03n03ooooo00Koo`03003ooooo083o
o`00Tooo00<00?P0ool01ooo00<00?ooool0P?oo002Cool00`00n03oo`07ool00`00ooooo`20ool0
09?oo`03n03ooooo00Ooo`80087oo`00Tooo00?h0?ooool01ooo00<00?ooool0P?oo002Bool00oP0
ooooo`08ool00`00ooooo`20ool009;oo`03n03ooooo00Soo`03003ooooo083oo`00TOoo00?h0000
ool02Ooo00<00?ooool0P?oo002Aool00oP0ooooo`09ool00`00ooooo`20ool008[oo`8000Coo`03
003h0?oo00Coo`03003ooooo00?oo`03003ooooo083oo`00ROoo00@00?ooool000?oo`03n03ooooo
00Coo`03003ooooo00?oo`03003ooooo083oo`00ROoo00@00?ooool000?oo`04n03oooooool50004
ool30020ool008Woo`04003ooooo0002ool00oP0ooooo`02ool01000ooooo`001Ooo00<00?ooool0
P?oo0029ool01000ooooo`000_oo00?h0?ooool00ooo00<00?oo00001Ooo00<00?ooool0P?oo0029
ool01`00ooooo`00ool00?P000Koo`8000Goo`03003ooooo083oo`00R_oo0P000_oo00?h0?ooool0
1_oo00<00?ooool00ooo00<00?ooool0P?oo002=ool00`00n03oo`0=ool00`00ooooo`20ool008go
o`03n03ooooo00goo`03003ooooo083oo`00S?oo00<00?P0ool03_oo00<00?ooool0P?oo002<ool0
0oP0ooooo`0>ool20021ool008_oo`03003h0?oo00ooo`03003ooooo083oo`00Rooo00?h0?ooool0
3ooo00<00?ooool0P?oo002:ool00`00n03oo`0@ool00`00ooooo`20ool008[oo`03n03ooooo013o
o`03003ooooo083oo`00ROoo00<00?P0ool04Ooo00<00?ooool0P?oo0029ool00oP0ooooo`0Aool0
0`00ooooo`20ool008Soo`03003h0?oo01;oo`03003ooooo083oo`00R?oo00?h0?ooool04_oo00<0
0?ooool0P?oo0027ool00`00n03oo`0Cool20021ool008Koo`03003h0?oo01Coo`03003ooooo083o
o`00Q_oo00?h0?ooool05?oo00<00?ooool0P?oo0025ool00`00n03oo`0Eool00`00ooooo`20ool0
08Coo`03003h0?oo01Koo`03003ooooo083oo`00Q?oo00?h0?ooool05_oo00<00?ooool0P?oo0020
ool5000Hool00`00ooooo`20ool0083oo`D001Soo`03003ooooo083oo`00Oooo1P006?oo0P00POoo
0020ool5000Hool00`00ooooo`20ool0083oo`D001Soo`03003ooooo083oo`00P?oo00?h0?oo0000
6_oo00<00?ooool0P?oo001oool00oP0ooooo`0Kool00`00ooooo`20ool007koo`03n0000?oo01co
o`03003ooooo083oo`00O_oo00?h0?ooool02Ooo0P001?oo00<00?ooool00ooo0P001Ooo00<00?oo
ool0P?oo001mool00oP0ooooo`09ool01000ooooo`002?oo00@00?ooool000Coo`03003ooooo083o
o`00O?oo00?h0000ool02_oo00@00?ooool000Soo`04003ooooo0004ool30020ool007coo`03n03o
oooo00[oo`04003ooooo0008ool30005ool00`00ooooo`20ool007_oo`03n03ooooo00_oo`04003o
oooo0008ool00`00ooooo`05ool00`00ooooo`20ool007[oo`03n03ooooo00coo`04003ooooo0008
ool00`00ooooo`05ool00`00ooooo`20ool007Woo`03003h0?oo00koo`8000[oo`<000Coo`03003o
oooo083oo`00NOoo00?h0?ooool08Ooo00<00?ooool0P?oo001hool00oP0ooooo`0Rool00`00oooo
o`20ool007Ooo`03003h0?oo02?oo`03003ooooo083oo`00M_oo00<00?P0ool09?oo00<00?ooool0
P?oo001eool00`00n03oo`0Uool20021ool007Coo`03003oooP002Koo`03003ooooo083oo`00Looo
00<00?oon0009ooo00<00?ooool0P?oo001bool00`00oooh000Xool00`00ooooo`20ool0077oo`03
003oooP002Woo`03003ooooo083oo`00L?oo00<00?oon000:_oo00<00?ooool0P?oo001_ool00`00
oooh000[ool00`00ooooo`20ool006koo`03003oooP002coo`03003ooooo083oo`00K_oo00<00?P0
ool0;?oo0P00POoo001]ool2n00^ool00`00ooooo`20ool006Soo`D0033oo`03003ooooo083oo`00
J?oo1@00<?oo00<00?ooool0P?oo001Wool6000`ool00`00ooooo`20ool006Soo`D0033oo`03003o
oooo083oo`00Iooo1P00<?oo00<00?ooool0P?oo001Vool01@00oooh0?oo0000<_oo00<00?ooool0
P?oo001Uool00`00oooh000eool20021ool006Coo`03003oooP003Koo`03003ooooo083oo`00Hooo
00<00?oon000=ooo00<00?ooool0P?oo001Rool00`00oooh000hool00`00ooooo`20ool0067oo`03
003oooP003Woo`03003ooooo083oo`00H?oo00<00?oon000>_oo00<00?ooool0P?oo001Nool20000
0ooon03oo`0jool00`00ooooo`20ool005goo`04003ooooon00Yool20004ool00`00ooooo`03ool2
0005ool00`00ooooo`20ool005_oo`8000;oo`03n03ooooo02Ooo`04003ooooo0008ool01000oooo
o`001?oo00<00?ooool0P?oo001Jool01@00oooooooon000:_oo00@00?ooool000Soo`04003ooooo
0004ool30020ool005Woo`05003ooooooooh000[ool01000ooooo`002Ooo0P001Ooo00<00?ooool0
P?oo001Gool20003ool00oP0ooooo`0Zool01000ooooo`002?oo00@00?ooool000Coo`03003ooooo
083oo`00E_oo00<00?ooool00_oo00?h0?ooool0:ooo00@00?ooool000Soo`04003ooooo0004ool0
0`00ooooo`20ool005Goo`04003oooooool2n00_ool2000:ool20005ool00`00ooooo`20ool005?o
o`8000;oo`;h04Coo`03003ooooo083oo`00C?oo1@0000Coo`00ooooo`;h04Koo`03003ooooo083o
o`00C?oo1P0000?oooP0n000B?oo00<00?ooool0P?oo001;ool60002n01:ool20021ool004coo`D0
04coo`03003ooooo083oo`00Booo1P00C?oo00<00?ooool0P?oo0019ool200001?oon03oo`00C_oo
00<00?ooool0P?oo0017ool20002ool00oP0ooooo`1?ool00`00ooooo`20ool004Goo`8000Coo`03
n03ooooo04ooo`03003ooooo083oo`00A?oo00<00?ooool00ooo00?h0?ooool0D?oo00<00?ooool0
P?oo0012ool20005ool00oP0ooooo`1Aool00`00ooooo`20ool0043oo`8000Koo`03n03ooooo05;o
o`80087oo`00?_oo0P001ooo00?h0?ooool0Dooo00<00?ooool0P?oo000kool30008ool00oP0oooo
o`1Dool00`00ooooo`20ool003Soo`<000Woo`;h05Ooo`03003ooooo083oo`00<?oo1@0000?oo`00
00002_oo0_P0FOoo00<00?ooool0P?oo000`ool6000;ool00oP0ooooo`1Iool00`00ooooo`20ool0
02ooo`H000[oo`;h05coo`03003ooooo083oo`00;Ooo20000oP00_oo1OP0G_oo00<00?ooool0P?oo
000Zool30003ool50002ool4n01Rool00`00ooooo`20ool002Ooo`<000Koo`03n03oo`0006[oo`80
087oo`009Ooo0P002Ooo00?h0?ooool0J_oo00<00?ooool0P?oo000Gool50004ool5000:ool00oP0
ooooo`1[ool00`00ooooo`20ool001Ooo`T000ooo`03n03ooooo06_oo`03003ooooo083oo`005_oo
1P004_oo00?h0?ooool0K?oo00<00?ooool0P?oo0008ool3000;ool6000Bool00oP0ooooo`1/ool0
0`00ooooo`20ool000Soo`<00003ool0000000P00003ool0000000<0017oo`03n03ooooo06Goo`@0
00Coo`03003ooooo083oo`002?oo10003?oo00?h0000n0004Ooo00?h0?ooool0J?oo00<00?ooool0
0ooo00<00?ooool0P?oo0008ool3000=ool01?P0ooooooP03ooo00?h0?ooool0JOoo00<00?ooool0
0ooo0`00P?oo0008ool3000=ool01OP0oooooooon0003Ooo00?h0?ooool0J_oo00<00?ooool00ooo
00<00?ooool0P?oo0008ool00`00ooooo`0<ool00oP0ooooo`03ool00oP0ooooo`09ool00oP0oooo
o`1[ool00`00ooooo`03ool00`00ooooo`20ool000Soo`;h00goo`03n03ooooo00Coo`;h00Ooo`;h
06coo`<000Goo`03003ooooo083oo`002?oo0_P03Ooo00?h0?ooool01_oo1oP0L?oo00<00?ooool0
0ooo00<00?ooool0P?oo0008ool2n00=ool00oP0ooooo`23ool00`00ooooo`20ool000Soo`;h00co
o`03n03ooooo08Coo`03003ooooo083oo`002?oo0_P03?oo00?h0?ooool0Q?oo00<00?ooool0P?oo
0008ool00oP0oooh000;ool00oP0ooooo`24ool20021ool000Soo`03n03oooP000_oo`03n03ooooo
08Coo`03003ooooo083oo`002?oo00?h0?oon0002_oo00?h0?ooool0QOoo00<00?ooool0P?oo0008
ool01?P0ooooooP02Ooo00?h0?ooool0QOoo00<00?ooool0P?oo0008ool01?P0ooooooP02Ooo00?h
0?ooool0QOoo00<00?ooool0P?oo0008ool01?P0ooooooP02Ooo00?h0?ooool0QOoo00<00?ooool0
P?oo0008ool01OP0oooooooon0001ooo00?h0?ooool0Q_oo00<00?ooool0P?oo0008ool01OP0oooo
oooon0001ooo00?h0?ooool0Q_oo00<00?ooool0P?oo0008ool00oP0ooooo`02ool00oP0ooooo`03
ool00oP0ooooo`27ool00`00ooooo`20ool000Soo`03n03ooooo00;oo`03n03ooooo00;oo`03n03o
oooo08Soo`80087oo`002?oo00?h0?ooool00ooo00Ch0?oooooh08_oo`03003ooooo083oo`002?oo
00?h0?ooool01?oo0oP0Rooo00<00?ooool0P?oo0008ool00oP0ooooo`2Bool00`00ooooo`20ool0
00Soo`03n03ooooo09;oo`03003ooooo083oo`002?oo00?h0?ooool0T_oo00<00?ooool0P?oo0008
ool00oP0ooooo`2Bool00`00ooooo`20ool000Soo`03n03ooooo09;oo`03003ooooo083oo`002?oo
00?h0?ooool0T_oo0P00POoo0008ool00oP0ooooo`2Bool00`00ooooo`20ool000Soo`03n03ooooo
09;oo`03003ooooo083oo`002?oo00?h0?ooool0T_oo00<00?ooool0P?oo0008ool00oP0ooooo`2B
ool00`00ooooo`20ool000Soo`03n03ooooo09;oo`03003ooooo083oo`002?oo00?h0?ooool0O_oo
10000ooo00<00?ooool00_oo10001?oo00<00?ooool0P?oo0008ool00oP0ooooo`20ool00`00oooo
o`07ool00`00ooooo`05ool00`00ooooo`20ool000Soo`03n03ooooo083oo`03003ooooo00Soo`03
003ooooo00Coo`<0083oo`002?oo00?h0?ooool0P?oo00<00?ooool02Ooo00<00?ooool00ooo00<0
0?ooool0P?oo0008ool00oP0ooooo`20ool00`00ooooo`0:ool00`00ooooo`02ool00`00ooooo`20
ool000Soo`03n03ooooo07koo`<000Woo`04003ooooo0004ool00`00ooooo`20ool000Soo`03n03o
oooo083oo`03003ooooo00Soo`8000Goo`03003ooooo083oo`002?oo00?h0?ooool0T_oo00<00?oo
ool0P?oo0008ool00oP0ooooo`2Bool00`00ooooo`20ool000Soo`<009;oo`03003ooooo083oo`00
2?oo0`00T_oo00<00?ooool0P?oo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {220.375, 0}} -> {0.050549, -0.0744195, \
0.00600988, 0.00600988}}]
}, Open  ]],

Cell["This adds a slight waviness to the ramp.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(wavyramp[x_]\  := \ 
        If[x\  < \ 1, \ 1\  - \ x\  + \ \  .05\ Cos[11\ Pi\ x]\ \ , 
          0];\)\), "\[IndentingNewLine]", 
    \(BouncingBall[ .75, \ wavyramp, {0, 1.25}]\)}], "Input",
  CellLabel->"In[43]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .67354 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
2.82977e-17 0.538834 1.44307e-16 0.538834 [
[.13471 -0.0125 -12 -9 ]
[.13471 -0.0125 12 0 ]
[.26942 -0.0125 -9 -9 ]
[.26942 -0.0125 9 0 ]
[.40413 -0.0125 -12 -9 ]
[.40413 -0.0125 12 0 ]
[.53883 -0.0125 -3 -9 ]
[.53883 -0.0125 3 0 ]
[.67354 -0.0125 -12 -9 ]
[.67354 -0.0125 12 0 ]
[.80825 -0.0125 -9 -9 ]
[.80825 -0.0125 9 0 ]
[.94296 -0.0125 -12 -9 ]
[.94296 -0.0125 12 0 ]
[-0.0125 .10777 -18 -4.5 ]
[-0.0125 .10777 0 4.5 ]
[-0.0125 .21553 -18 -4.5 ]
[-0.0125 .21553 0 4.5 ]
[-0.0125 .3233 -18 -4.5 ]
[-0.0125 .3233 0 4.5 ]
[-0.0125 .43107 -18 -4.5 ]
[-0.0125 .43107 0 4.5 ]
[-0.0125 .53883 -6 -4.5 ]
[-0.0125 .53883 0 4.5 ]
[-0.0125 .6466 -18 -4.5 ]
[-0.0125 .6466 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .67354 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.13471 0 m
.13471 .00625 L
s
[(0.25)] .13471 -0.0125 0 1 Mshowa
.26942 0 m
.26942 .00625 L
s
[(0.5)] .26942 -0.0125 0 1 Mshowa
.40413 0 m
.40413 .00625 L
s
[(0.75)] .40413 -0.0125 0 1 Mshowa
.53883 0 m
.53883 .00625 L
s
[(1)] .53883 -0.0125 0 1 Mshowa
.67354 0 m
.67354 .00625 L
s
[(1.25)] .67354 -0.0125 0 1 Mshowa
.80825 0 m
.80825 .00625 L
s
[(1.5)] .80825 -0.0125 0 1 Mshowa
.94296 0 m
.94296 .00625 L
s
[(1.75)] .94296 -0.0125 0 1 Mshowa
.125 Mabswid
.02694 0 m
.02694 .00375 L
s
.05388 0 m
.05388 .00375 L
s
.08083 0 m
.08083 .00375 L
s
.10777 0 m
.10777 .00375 L
s
.16165 0 m
.16165 .00375 L
s
.18859 0 m
.18859 .00375 L
s
.21553 0 m
.21553 .00375 L
s
.24248 0 m
.24248 .00375 L
s
.29636 0 m
.29636 .00375 L
s
.3233 0 m
.3233 .00375 L
s
.35024 0 m
.35024 .00375 L
s
.37718 0 m
.37718 .00375 L
s
.43107 0 m
.43107 .00375 L
s
.45801 0 m
.45801 .00375 L
s
.48495 0 m
.48495 .00375 L
s
.51189 0 m
.51189 .00375 L
s
.56578 0 m
.56578 .00375 L
s
.59272 0 m
.59272 .00375 L
s
.61966 0 m
.61966 .00375 L
s
.6466 0 m
.6466 .00375 L
s
.70048 0 m
.70048 .00375 L
s
.72743 0 m
.72743 .00375 L
s
.75437 0 m
.75437 .00375 L
s
.78131 0 m
.78131 .00375 L
s
.83519 0 m
.83519 .00375 L
s
.86213 0 m
.86213 .00375 L
s
.88908 0 m
.88908 .00375 L
s
.91602 0 m
.91602 .00375 L
s
.9699 0 m
.9699 .00375 L
s
.99684 0 m
.99684 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .10777 m
.00625 .10777 L
s
[(0.2)] -0.0125 .10777 1 0 Mshowa
0 .21553 m
.00625 .21553 L
s
[(0.4)] -0.0125 .21553 1 0 Mshowa
0 .3233 m
.00625 .3233 L
s
[(0.6)] -0.0125 .3233 1 0 Mshowa
0 .43107 m
.00625 .43107 L
s
[(0.8)] -0.0125 .43107 1 0 Mshowa
0 .53883 m
.00625 .53883 L
s
[(1)] -0.0125 .53883 1 0 Mshowa
0 .6466 m
.00625 .6466 L
s
[(1.2)] -0.0125 .6466 1 0 Mshowa
.125 Mabswid
0 .02694 m
.00375 .02694 L
s
0 .05388 m
.00375 .05388 L
s
0 .08083 m
.00375 .08083 L
s
0 .13471 m
.00375 .13471 L
s
0 .16165 m
.00375 .16165 L
s
0 .18859 m
.00375 .18859 L
s
0 .24248 m
.00375 .24248 L
s
0 .26942 m
.00375 .26942 L
s
0 .29636 m
.00375 .29636 L
s
0 .35024 m
.00375 .35024 L
s
0 .37718 m
.00375 .37718 L
s
0 .40413 m
.00375 .40413 L
s
0 .45801 m
.00375 .45801 L
s
0 .48495 m
.00375 .48495 L
s
0 .51189 m
.00375 .51189 L
s
0 .56578 m
.00375 .56578 L
s
0 .59272 m
.00375 .59272 L
s
0 .61966 m
.00375 .61966 L
s
.25 Mabswid
0 0 m
0 .67354 L
s
0 0 m
1 0 L
1 .67354 L
0 .67354 L
closepath
clip
newpath
.5 Mabswid
0 .56578 m
.00123 .56447 L
.00245 .56299 L
.00491 .55954 L
.00981 .5508 L
.01472 .53991 L
.01963 .52747 L
.03027 .49881 L
.03559 .48565 L
.04091 .47452 L
.04339 .47021 L
.04588 .46655 L
.04836 .46355 L
.05084 .46124 L
.05208 .46034 L
.05333 .4596 L
.05457 .45903 L
.05581 .45862 L
.05705 .45837 L
.05829 .45826 L
.05954 .45829 L
.06078 .45846 L
.062 .45874 L
.06321 .45914 L
.06443 .45963 L
.06565 .46022 L
.06808 .46161 L
.07052 .46323 L
.07539 .46674 L
.07782 .46842 L
.08026 .46993 L
.08158 .47063 L
.0829 .47124 L
.08422 .47174 L
.08554 .47212 L
.08686 .47236 L
.08818 .47246 L
.0895 .4724 L
.09082 .47217 L
.09214 .47177 L
.09346 .47119 L
.09479 .47043 L
.09611 .46948 L
.09743 .46833 L
.09875 .46699 L
.10139 .46374 L
.10385 .46003 L
.10632 .45569 L
.11125 .44533 L
.12111 .42006 L
Mistroke
.13179 .39186 L
.13714 .37992 L
.13981 .37486 L
.14248 .37051 L
.1451 .36698 L
.14641 .36549 L
.14773 .3642 L
.14904 .36309 L
.15035 .36218 L
.15166 .36145 L
.15297 .3609 L
.15428 .36053 L
.15559 .36033 L
.15691 .36029 L
.15822 .3604 L
.15953 .36066 L
.16084 .36105 L
.16215 .36156 L
.16346 .36217 L
.16469 .36283 L
.16591 .36356 L
.16836 .36517 L
.17325 .36869 L
.1757 .37039 L
.17692 .37118 L
.17814 .37191 L
.18059 .37315 L
.18181 .37364 L
.18303 .37403 L
.18436 .37432 L
.18569 .37447 L
.18701 .37447 L
.18834 .3743 L
.18967 .37396 L
.19099 .37344 L
.19232 .37273 L
.19365 .37184 L
.19497 .37075 L
.1963 .36947 L
.19895 .36632 L
.2016 .36241 L
.20426 .35777 L
.20673 .35284 L
.20921 .34738 L
.21416 .33521 L
.22407 .30853 L
.22893 .29595 L
.23378 .2847 L
.23864 .27542 L
.24107 .27165 L
Mistroke
.24349 .26851 L
.24613 .26584 L
.24744 .2648 L
.24876 .26394 L
.25008 .26327 L
.2514 .26278 L
.25271 .26247 L
.25403 .26233 L
.25535 .26234 L
.25666 .26251 L
.25798 .26282 L
.2593 .26325 L
.26062 .26381 L
.26193 .26446 L
.26457 .26602 L
.26702 .26769 L
.26948 .26946 L
.27194 .27123 L
.2744 .2729 L
.27686 .27435 L
.27809 .27497 L
.27931 .2755 L
.28054 .27593 L
.28177 .27625 L
.283 .27645 L
.28423 .27652 L
.28556 .27644 L
.2869 .2762 L
.28823 .27578 L
.28956 .27517 L
.29089 .27438 L
.29222 .27339 L
.29356 .27221 L
.29489 .27083 L
.29622 .26926 L
.29755 .26749 L
.30022 .26338 L
.30555 .25307 L
.31078 .24071 L
.31601 .22695 L
.32124 .21272 L
.32647 .19903 L
.33135 .1876 L
.33379 .18257 L
.33623 .17809 L
.33867 .17421 L
.33989 .1725 L
.34111 .17096 L
.34355 .16837 L
.34477 .16732 L
Mistroke
.34599 .16643 L
.34731 .16566 L
.34863 .16506 L
.34995 .16465 L
.35128 .16442 L
.3526 .16435 L
.35392 .16443 L
.35525 .16467 L
.35657 .16504 L
.35789 .16553 L
.35921 .16614 L
.36186 .16762 L
.36715 .17128 L
.37209 .17475 L
.37456 .17624 L
.37579 .17687 L
.37703 .17742 L
.37826 .17788 L
.3795 .17822 L
.38073 .17845 L
.38197 .17855 L
.3832 .17851 L
.38444 .17833 L
.38567 .178 L
.38691 .17751 L
.38825 .1768 L
.38958 .1759 L
.39092 .1748 L
.39226 .17351 L
.39493 .17032 L
.39761 .16636 L
.40296 .15629 L
.40831 .14383 L
.41882 .11579 L
.42408 .10198 L
.42933 .0896 L
.43423 .08006 L
.43668 .07617 L
.43913 .07292 L
.44159 .07034 L
.44281 .06929 L
.44404 .06841 L
.44526 .06769 L
.44649 .06714 L
.44771 .06674 L
.44894 .06649 L
.45027 .06638 L
.4516 .06643 L
.45292 .06663 L
.45425 .06698 L
Mistroke
.45558 .06745 L
.45691 .06803 L
.45957 .06949 L
.46222 .07123 L
.46488 .07313 L
.4702 .07687 L
.47268 .07835 L
.47392 .07898 L
.47516 .07952 L
.4764 .07996 L
.47764 .08029 L
.47888 .0805 L
.48012 .08058 L
.48136 .08052 L
.4826 .08032 L
.48384 .07996 L
.48508 .07944 L
.48632 .07877 L
.48756 .07792 L
.4888 .07691 L
.49004 .07573 L
.49126 .0744 L
.49248 .07292 L
.49491 .06946 L
.49977 .06073 L
.50464 .0499 L
.5095 .03756 L
.52006 .00912 L
Mfstroke
.52006 .00912 m
.52374 0 L
s
.53923 0 m
.53923 0 L
.54046 0 L
.54169 0 L
.54292 0 L
.54539 0 L
.55031 0 L
.56098 0 L
.57166 0 L
.59262 0 L
.61217 0 L
.63337 0 L
.65316 0 L
.67256 0 L
.69362 0 L
.71326 0 L
.73455 0 L
.75443 0 L
.77393 0 L
.79507 0 L
.81481 0 L
.83619 0 L
.85719 0 L
.87677 0 L
.89801 0 L
.91783 0 L
.93727 0 L
.95836 0 L
.97803 0 L
.99936 0 L
1 0 L
s
1 0 0 r
0 .67354 m
0 .67352 L
0 .67344 L
0 .67313 L
0 .67191 L
0 .667 L
0 .64511 L
0 .61078 L
0 .58979 L
0 .57816 L
0 .57206 L
0 .56894 L
0 .56578 L
s
0 .56578 m
.00249 .56575 L
.00498 .56567 L
.00996 .56537 L
.01992 .56414 L
.03984 .55923 L
.08303 .53734 L
.12336 .50301 L
.1629 .45633 L
.20579 .39111 L
.24582 .31656 L
.25711 .29314 L
.26276 .28104 L
.26558 .27489 L
.2684 .26868 L
s
.2684 .26868 m
.2631 .27347 L
.2578 .27813 L
.2472 .28706 L
.23661 .29549 L
.23131 .29951 L
.22601 .3034 L
s
.22601 .3034 m
.22667 .30639 L
.22733 .30934 L
.22866 .31508 L
.23131 .32595 L
.23661 .34522 L
.24811 .37578 L
.25347 .38477 L
.25616 .38801 L
.25884 .39041 L
.26147 .39196 L
.26279 .39242 L
.2641 .39269 L
.26476 .39275 L
.26542 .39275 L
.26607 .39271 L
.26673 .39262 L
.26805 .39228 L
.26936 .39174 L
.27507 .38707 L
.28078 .3786 L
.29143 .35264 L
.30297 .30959 L
.31431 .25223 L
.31739 .23408 L
.31892 .22458 L
.31969 .21974 L
.32046 .21482 L
s
.32046 .21482 m
.32211 .21212 L
.32376 .20938 L
.32705 .20375 L
.33363 .19191 L
.34021 .17931 L
.34351 .17272 L
.34515 .16935 L
.3468 .16593 L
s
.3468 .16593 m
.3501 .16618 L
.35341 .16637 L
.35671 .16649 L
.36001 .16655 L
s
.36001 .16655 m
.36124 .16836 L
.36247 .17011 L
.36493 .17347 L
.36984 .17957 L
.37967 .18931 L
.39032 .19618 L
.39565 .19817 L
.39831 .1988 L
.40098 .19919 L
.40222 .1993 L
.40346 .19935 L
.40471 .19934 L
.40595 .19929 L
.40844 .19902 L
.41092 .19854 L
.42087 .19453 L
.44038 .17694 L
.46154 .14327 L
.48129 .09816 L
.4852 .08766 L
.48715 .08221 L
.48911 .07664 L
s
.48911 .07664 m
.49183 .07592 L
.49456 .07515 L
.50001 .07346 L
.51092 .06947 L
.53273 .05903 L
.58001 .02515 L
.593 .01315 L
.59949 .00672 L
.60273 .0034 L
.60598 0 L
s
.60598 0 m
.60802 .00214 L
.61007 .00422 L
.61416 .00824 L
.62233 .01567 L
.63869 .02806 L
.65642 .0378 L
.66529 .04123 L
.67416 .04369 L
.68243 .04512 L
.68657 .04553 L
.68864 .04565 L
.69071 .04572 L
.69278 .04574 L
.69485 .0457 L
.69692 .04561 L
.69899 .04547 L
.70727 .04439 L
.7235 .03984 L
.73974 .03206 L
.77495 .00409 L
.77693 .00207 L
.77891 0 L
s
.77891 0 m
.78044 .0016 L
.78198 .00314 L
.78504 .00608 L
.79118 .01134 L
.80345 .01941 L
.81675 .02446 L
.82007 .02512 L
.82173 .02536 L
.82339 .02554 L
.82506 .02566 L
.82672 .02572 L
.82838 .02572 L
.83004 .02566 L
.8316 .02555 L
.83315 .02539 L
.83625 .0249 L
.84246 .02331 L
.85488 .0176 L
.86553 .01004 L
.87086 .00533 L
.87352 .00274 L
.87485 .00139 L
.87619 0 L
s
.87619 0 m
.87734 .00119 L
.87849 .00233 L
.88079 .00446 L
.88539 .0081 L
.88999 .01092 L
.89459 .01292 L
.89957 .01417 L
.90082 .01433 L
.90207 .01443 L
.90331 .01447 L
.90456 .01445 L
.90581 .01437 L
.90706 .01423 L
.90955 .01377 L
.91454 .01214 L
.92272 .00736 L
.92681 .004 L
.92886 .00208 L
.92988 .00106 L
.9309 0 L
s
.9309 0 m
.93177 .00089 L
.93263 .00172 L
.93435 .00324 L
.9378 .00566 L
.94125 .00727 L
.94298 .00776 L
.9447 .00805 L
.94564 .00813 L
.94657 .00814 L
.94751 .00809 L
.94844 .00798 L
.95031 .00758 L
.95218 .00695 L
.95967 .00199 L
.96067 .00103 L
.96168 0 L
s
.96168 0 m
.96233 .00066 L
.96298 .00127 L
.96427 .00233 L
.96686 .00384 L
.96815 .00429 L
.9688 .00443 L
.96944 .00453 L
.97009 .00458 L
.97074 .00457 L
.97139 .00451 L
.97203 .0044 L
.9729 .00418 L
.97377 .00386 L
.97551 .00294 L
.97725 .00166 L
.97812 .00087 L
.97899 0 L
s
.97899 0 m
.97948 .00049 L
.97996 .00092 L
.98093 .00164 L
.98191 .00216 L
.98288 .00247 L
.98336 .00255 L
.98385 .00258 L
.98433 .00255 L
.98482 .00248 L
.98579 .00217 L
.98676 .00167 L
.98774 .00094 L
.98824 .0005 L
.98873 0 L
s
.98873 0 m
.98907 .00034 L
.98942 .00063 L
.9901 .00109 L
.99079 .00136 L
.99113 .00143 L
.99147 .00145 L
.99181 .00143 L
.99216 .00136 L
.99284 .00109 L
.99353 .00063 L
.99387 .00034 L
.99421 0 L
s
.99421 0 m
.9946 .00036 L
.99498 .00061 L
.99537 .00076 L
.99575 .00081 L
.99614 .00076 L
.99652 .00061 L
.99691 .00036 L
.99729 0 L
s
.99729 0 m
.99751 .0002 L
.99773 .00034 L
.99794 .00043 L
.99816 .00046 L
.99838 .00043 L
.99859 .00034 L
.99881 .0002 L
.99903 0 L
s
.99903 0 m
.99927 .00019 L
.99951 .00026 L
.99976 .00019 L
1 0 L
s
0 g
.025 w
0 .67354 Mdot
0 .56578 Mdot
.2684 .26868 Mdot
.22601 .3034 Mdot
.32046 .21482 Mdot
.3468 .16593 Mdot
.36001 .16655 Mdot
.48911 .07664 Mdot
.60598 0 Mdot
.77891 0 Mdot
.87619 0 Mdot
.9309 0 Mdot
.96168 0 Mdot
.97899 0 Mdot
.98873 0 Mdot
.99421 0 Mdot
.99729 0 Mdot
.99903 0 Mdot
1 0 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[43]:=",
  ImageSize->{288, 193.938},
  ImageMargins->{{1, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000`A000`40n007h00Ooooo8Ooo003ooolQ
ool002ooo`8000Coo`03003ooooo00;oo`@000;oo`<001;oo`8000Coo`03003ooooo00;oo`<001?o
o`8000Coo`03003ooooo00?oo`03003ooooo00;oo`<001Soo`@001Ooo`@000?oo`03003ooooo00;o
o`@000;oo`<001;oo`@000?oo`03003ooooo00;oo`<0017oo`@000?oo`03003ooooo00?oo`03003o
oooo00;oo`<000Ooo`00;_oo00@00?ooool000Soo`03003ooooo00Koo`03003ooooo00koo`04003o
oooo000;ool00`00ooooo`0?ool01000ooooo`002Ooo00<00?ooool01Ooo00<00?ooool05ooo00<0
0?ooool06?oo00<00?ooool01ooo00<00?ooool01_oo00<00?ooool04Ooo00<00?ooool02_oo00<0
0?ooool04?oo00<00?ooool02?oo00<00?ooool01Ooo00<00?ooool01?oo000^ool01000ooooo`00
2Ooo00<00?ooool01Ooo00<00?ooool03_oo00@00?ooool000_oo`03003ooooo00ooo`04003ooooo
000:ool00`00ooooo`04ool00`00ooooo`0Gool00`00ooooo`0Hool00`00ooooo`08ool00`00oooo
o`05ool00`00ooooo`0Aool00`00ooooo`0:ool00`00ooooo`0@ool00`00ooooo`09ool00`00oooo
o`04ool00`00ooooo`04ool002koo`04003ooooo000:ool01000oooooooo0`004Ooo00@00?ooool0
00Soo`<001;oo`04003ooooo000:ool01000oooooooo0`006_oo00<00?ooool06?oo00<00?ooool0
2Ooo00@00?ooooooo`<001Coo`03003ooooo00Ooo`<001?oo`03003ooooo00Woo`04003oooooool3
0007ool002koo`04003ooooo000;ool01000ooooo`004ooo00@00?ooool000Soo`03003ooooo01;o
o`04003ooooo000;ool01000ooooo`007?oo00<00?ooool06?oo00<00?ooool02_oo00@00?ooool0
01Koo`03003ooooo00Ooo`03003ooooo01?oo`03003ooooo00[oo`04003ooooo0009ool002koo`04
003ooooo0008ool01000ooooo`000_oo00<00?ooool04Ooo00@00?ooool000Soo`03003ooooo01;o
o`04003ooooo0008ool01000ooooo`000_oo00<00?ooool06?oo0`006?oo0`002Ooo00@00?ooool0
00;oo`03003ooooo01;oo`<000Woo`03003ooooo017oo`<000Woo`04003ooooo0002ool00`00oooo
o`07ool002ooo`8000[oo`8000?oo`@0017oo`8000Woo`@001;oo`8000Woo`@000;oo`@001Woo`03
003ooooo01Soo`03003ooooo00Soo`8000?oo`@001?oo`03003ooooo00Ooo`@001;oo`03003ooooo
00Ooo`@000;oo`@000Koo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo000Eoooo000<0000
01Goo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo
00Goo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo
00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo00Coo`03003ooooo
00Coo`03003ooooo00Coo`03003ooooo00Goo`03003ooooo00Coo`03003ooooo00Coo`04003ooooo
0003ool00`00ooooo`04ool00`00ooooo`04ool00`00ool000050007ool00`00ooooo`04ool00`00
ooooo`04ool00`00ooooo`04ool00`00ooooo`04ool00`00ooooo`02ool50005ool00`00ooooo`04
ool00`00ooooo`04ool00`00ool000050006ool00`00ool000050003ool=000001Goo`03003ooooo
08Ooo`03003ooooo017oo`03n000000000<002Soo`03n000000000<001Coo`03n000000000<000So
o`03n000000000<000Ch00d000005Ooo00<00?ooool0Qooo00<00?ooool04?oo00Gh0?ooooooo`00
00;oo`03n03ooooo02Coo`05n03oooooool00002ool00oP0ooooo`0?ool2n003ool00`00ooooo`04
n0001?oon03h0?P00ooo00<00?ooool01_oo00D00?ooooooo`0000;oo`@000005Ooo00<00?ooool0
Q_oo00<00?ooool04?oo00?h0?ooool01_oo00?h0?ooool08Ooo0_P02?oo0_P03_oo00?h0?ooool0
2Ooo0_P06_oo000Eool00`00ooooo`26ool00`00ooooo`0?ool00oP0ooooo`08ool2n00Pool00oP0
ooooo`0:ool2n00:ool2n00Xool001Goo`03003ooooo08Koo`03003ooooo00koo`03n03ooooo00_o
o`;h01goo`03n03ooooo00goo`Ch00;oo`Ch02[oo`005Ooo0P00Q_oo00<00?ooool03Ooo0_P04?oo
00?h0?ooool06?oo0_P05?oo0oP0;Ooo000Eool00`00ooooo`25ool00`00ooooo`0<ool00oP0oooo
o`0Aool00oP0ooooo`0Eool2n016ool001Goo`03003ooooo08Coo`03003ooooo00_oo`;h01Goo`;h
01?oo`;h04Soo`005Ooo00<00?ooool0Q?oo00<00?ooool02_oo00?h0?ooool05ooo0_P03ooo0_P0
B_oo000Eool00`00ooooo`24ool00`00ooooo`09ool00oP0ooooo`0Jool4n009ool2n01<ool001Go
o`03003ooooo08?oo`03003ooooo00Soo`;h027oo`Wh04koo`005Ooo00<00?ooool0Pooo00<00?oo
ool01ooo00?h0?ooool0N?oo000Eool20024ool00`00ooooo`05ool2n01kool001Goo`03003ooooo
08;oo`03003ooooo00Goo`03n03ooooo07_oo`005Ooo00<00?ooool0P_oo00<00?ooool00ooo0_P0
O_oo000Eool00`00ooooo`1dool2000;ool00`00ooooo`02ool2n020ool001Goo`03003ooooo07;o
o`8000;oo`8000Goo`D000;oo`;h08;oo`005Ooo00<00?ooool0LOoo00<00?ooool01?oo00<00?oo
ool00_oo1@000_P0Q?oo000Eool00`00ooooo`1aool00`00ooooo`05ool200000ooo000000040026
ool001Goo`80077oo`03003ooooo00Ooo`P008Koo`005Ooo00<00?ooool0L?oo00<00?ooool02_oo
1@00Q_oo000Eool00`00ooooo`1_ool00`00ooooo`0<ool00oP0003oo`27ool001Goo`03003ooooo
06ooo`03003ooooo00coo`03n03ooooo08Ooo`005Ooo00<00?ooool0K_oo00<00?ooool03?oo00?h
0?ooool0R?oo0002ool20004ool00`00ooooo`02ool40004ool00`00ooooo`1^ool00`00ooooo`0<
ool00oP0ooooo`28ool00005ool00?ooool00008ool00`00ooooo`05ool00`00ooooo`1]ool00`00
ooooo`0=ool00oP0ooooo`28ool00005ool00?ooool00009ool00`00ooooo`04ool2001^ool00`00
ooooo`0<ool00oP0ooooo`29ool00005ool00?ooool0000:ool00`00ooooo`03ool00`00ooooo`1/
ool00`00ooooo`0=ool00oP0ooooo`29ool00005ool00?ooool0000;ool00`00ooooo`02ool00`00
ooooo`1/ool00`00ooooo`0<ool00oP0ooooo`2:ool00005ool00?ooool00008ool01000ooooo`00
1?oo00<00?ooool0K?oo00<00?ooool03?oo00?h0?ooool0R_oo0002ool2000:ool20005ool00`00
ooooo`1[ool00`00ooooo`0<ool00oP0ooooo`2;ool001Goo`03003ooooo06_oo`03003ooooo00co
o`03n03ooooo08_oo`005Ooo00<00?ooool0Jooo00<00?ooool02ooo00?h0?ooool0S?oo000Eool0
0`00ooooo`1Zool00`00ooooo`0<ool00oP0ooooo`2<ool001Goo`8006_oo`03003ooooo00_oo`03
n03ooooo08goo`005Ooo00<00?ooool0JOoo00<00?ooool03?oo00?h0?ooool0SOoo000Eool00`00
ooooo`1Yool00`00ooooo`0;ool00oP0ooooo`2>ool001Goo`03003ooooo06Woo`03003ooooo00[o
o`03n03ooooo08ooo`005Ooo00<00?ooool0J?oo00<00?ooool02ooo00?h0?ooool0Sooo000Eool0
0`00ooooo`1Xool00`00ooooo`0:ool00oP0ooooo`2@ool001Goo`03003ooooo05Ooo`T000Ooo`03
003ooooo00_oo`03n03ooooo093oo`005Ooo0P00F?oo2@001ooo00<00?ooool02_oo00?h0?ooool0
TOoo000Eool00`00ooooo`1Fool:0006ool00`00ooooo`0;ool00oP0ooooo`2Aool001Goo`03003o
oooo05Ooo`T000Koo`03003ooooo00[oo`03n03ooooo09;oo`005Ooo00<00?ooool0Eooo2P000ooo
0P003Ooo00?h0?ooool0T_oo000Eool00`00ooooo`1Fool01000n03oo`000ooo00<00?oon0001@00
3Ooo00?h0?ooool0Tooo000Eool00`00ooooo`1Fool00`00n03oo`07ool00oP0ooooo`0>ool00oP0
ooooo`2Dool001Goo`03003ooooo05Koo`03n03ooooo00Soo`03n03ooooo00_oo`;h09Ooo`005Ooo
0P00E_oo00<00?P0ool02_oo00?h0?ooool02Ooo00?h0?ooool0Uooo000Eool00`00ooooo`1Eool0
0oP0ooooo`0;ool2n007ool2n02Jool001Goo`03003ooooo05Coo`03003h0?oo00koo`;h00?oo`;h
09coo`005Ooo00<00?ooool0E?oo00?h0?ooool04?oo0oP0W_oo000Eool00`00ooooo`1Cool00`00
n03oo`2bool000;oo`8000Coo`03003ooooo00Coo`03003ooooo00?oo`03003ooooo053oo`D00;?o
o`0000Goo`00ooooo`0000[oo`03003ooooo00?oo`03003ooooo053oo`D00;?oo`0000Goo`00oooo
o`0000Ooo`D000Coo`80053oo`H00;?oo`0000Goo`00ooooo`0000Ooo`04003ooooo0005ool00`00
ooooo`1@ool5002cool00005ool00?ooool00008ool00`00ool00005ool00`00ooooo`1@ool5002c
ool00005ool00?ooool00009ool20005ool00`00ooooo`1Aool2002eool000;oo`8000_oo`03003o
oooo00?oo`03003ooooo057oo`03n03ooooo0;Coo`005Ooo00<00?ooool0D?oo00<00?P0ool0]Ooo
000Eool00`00ooooo`1@ool00`00n03oo`2eool001Goo`80053oo`03003oooP00;Koo`005Ooo00<0
0?ooool0Cooo00<00?P0ool0]_oo000Eool00`00ooooo`1>ool00`00oooh002gool001Goo`03003o
oooo04koo`03003oooP00;Ooo`005Ooo00<00?ooool0C_oo00<00?oon000]ooo000Eool00`00oooo
o`10ool70006ool01000ooooooP0]ooo000Eool00`00ooooo`0nool90006ool00`00oooh002hool0
01Goo`8003ooo`03003ooooo00H000Goo`04003ooooon02hool001Goo`03003ooooo03goo`03003o
oooo00;oo`D000?oo`8000?oo`03n03ooooo0;Koo`005Ooo00<00?ooool0??oo00<00?ooool00ooo
2@001?oo00?h0?ooool0]_oo000Eool00`00ooooo`0lool00`00ooooo`02ool01?P0oooh00002_oo
00?h0?ooool0]_oo000Eool00`00ooooo`0kool01000oooooooo0_P00_oo00?h0?ooool02?oo00?h
0?ooool0]ooo000Eool00`00ooooo`0kool01000ooooooP00ooo00?h0?ooool02Ooo00?h0?ooool0
]ooo000Eool00`00ooooo`0jool01000ooooooP01?oo00?h0?ooool02Ooo00?h0?ooool0]ooo000E
ool2000hool500000oP0ooooo`02ool00oP0ooooo`0:ool00oP0ooooo`2gool001Goo`03003ooooo
03Ooo`D000Goo`03n03ooooo00[oo`03n03ooooo0;Ooo`005Ooo00<00?ooool0=_oo1P001?oo00?h
0?ooool02_oo00?h0?ooool0^?oo000Eool00`00ooooo`0gool50004ool00oP0ooooo`0:ool00oP0
ooooo`2hool001Goo`03003ooooo03Ooo`D000?oo`03n03ooooo00_oo`03n03ooooo0;Soo`005Ooo
00<00?ooool0>?oo0P0000?h0?ooool00_oo00?h0?ooool02ooo00?h0?ooool0^?oo0002ool20004
ool00`00ooooo`03ool20005ool00`00ooooo`0gool01000ooooooP00ooo00?h0?ooool02ooo00?h
0?ooool0^Ooo00001Ooo003ooooo00002?oo00@00?ooool000Coo`03003ooooo03Ooo`04003ooooo
n002ool00oP0ooooo`0<ool00oP0ooooo`2iool00005ool00?ooool00008ool01000ooooo`001?oo
0P00>?oo00@00?oooooh00;oo`03n03ooooo00coo`03n03ooooo0;Woo`0000Goo`00ooooo`0000So
o`<000Goo`03003ooooo03Ooo`06003ooooon03oooP03ooo00?h0?ooool0^Ooo00001Ooo003ooooo
00002?oo00<00?ooool01Ooo00<00?ooool0=_oo00<00?ooool00_oo0_P03_oo00?h0?ooool0^_oo
00001Ooo003ooooo00002?oo00<00?ooool01Ooo00<00?ooool0=_oo00<00?ooool00_oo00?h0?oo
ool03Ooo00?h0?ooool0^_oo0002ool2000:ool30004ool00`00ooooo`0eool00`00ooooo`03ool2
n00>ool00oP0ooooo`2jool001Goo`03003ooooo03Goo`03003ooooo00;oo`03n03oooP000koo`03
n03ooooo0;[oo`005Ooo00<00?ooool0=?oo00<00?ooool00ooo00?h0?oon0003Ooo00?h0?ooool0
^ooo000Eool2000eool00`00ooooo`02ool01OP0oooooooon0003?oo00?h0?ooool0^ooo000Eool0
0`00ooooo`0dool00`00ooooo`02ool01OP0oooooooon0003?oo00?h0?ooool0^ooo000Eool00`00
ooooo`0Vool2000;ool00`00ooooo`02ool00oP0ooooo`02ool00oP0ooooo`09ool00oP0ooooo`2l
ool001Goo`03003ooooo02Coo`H000Soo`03003ooooo00;oo`03n03ooooo00Coo`03n03ooooo00So
o`03n03ooooo0;coo`005Ooo00<00?ooool08ooo0P001Ooo00<00?ooool01Ooo00<00?ooool00_oo
00?h0?ooool01?oo00?h0?ooool01ooo00?h0?ooool0_Ooo000Eool00`00ooooo`0Sool00`00oooo
o`05ool20004ool00`00ooooo`02ool00oP0ooooo`05ool00oP0ooooo`07ool00oP0ooooo`2mool0
01Goo`03003ooooo02;oo`03003ooooo00Ooo`D000Goo`03n03ooooo00Koo`03n03ooooo00Goo`03
n03ooooo0;koo`005Ooo0P008ooo00<00?ooool04?oo00?h0?ooool01ooo00?h0?ooool01Ooo00?h
0?ooool0__oo000Eool00`00ooooo`0Qool00`00ooooo`0Aool00oP0ooooo`07ool00oP0ooooo`05
ool00oP0ooooo`2nool001Goo`03003ooooo027oo`03003ooooo013oo`03n03ooooo00Woo`03n03o
oooo00?oo`03n03ooooo0;ooo`005Ooo00<00?ooool08?oo00<00?ooool04Ooo00?h0?ooool02_oo
0_P00_oo0_P0`Ooo000Eool00`00ooooo`0Pool00`00ooooo`0@ool00oP0ooooo`0<ool3n033ool0
01Goo`03003ooooo023oo`03003ooooo00ooo`03n03ooooo0=?oo`005Ooo00<00?ooool07ooo00<0
0?ooool04?oo00?h0?ooool0dooo000Eool2000Pool00`00ooooo`0?ool00oP0ooooo`3Dool001Go
o`03003ooooo01koo`03003ooooo00ooo`03n03ooooo0=Goo`005Ooo00<00?ooool07_oo00<00?oo
ool03ooo00?h0?ooool0eOoo000Eool00`00ooooo`0Mool00`00ooooo`0?ool00oP0ooooo`3Fool0
01Goo`03003ooooo01goo`03003ooooo00koo`03n03ooooo0=Ooo`000_oo0P001?oo00<00?ooool0
0ooo0P001Ooo00<00?ooool07Ooo00<00?ooool03_oo00?h0?ooool0eooo00001Ooo003ooooo0000
2?oo00@00?ooool000Coo`03003ooooo01coo`03003ooooo00koo`03n03ooooo0=Soo`0000Goo`00
ooooo`0000Soo`04003ooooo0004ool2000Mool00`00ooooo`0>ool00oP0ooooo`3Hool00005ool0
0?ooool00009ool20005ool00`00ooooo`0Kool00`00ooooo`0>ool00oP0ooooo`3Iool00005ool0
0?ooool00008ool01000ooooo`001?oo00<00?ooool06ooo00<00?ooool03Ooo00?h0?ooool0f_oo
00001Ooo003ooooo00002?oo00@00?ooool000Coo`03003ooooo01[oo`03003ooooo00koo`03n03o
oooo0=[oo`000_oo0P002_oo0P001Ooo00<00?ooool06_oo00<00?ooool03Ooo00?h0?ooool0fooo
000Eool00`00ooooo`0Jool00`00ooooo`0<ool00oP0ooooo`3Lool001Goo`03003ooooo01Woo`03
003ooooo00goo`03n03ooooo0=coo`005Ooo0P003Ooo0P002ooo00<00?ooool03?oo00?h0?ooool0
gOoo000Eool00`00ooooo`0:ool60008ool00`00ooooo`0<ool00oP0ooooo`3Nool001Goo`03003o
oooo00[oo`03003ooooo00?oo`03003ooooo00Goo`03003ooooo00_oo`03n03ooooo0=ooo`005Ooo
00<00?ooool02Ooo00<00?ooool01Ooo00<00?ooool00_oo0P003_oo00?h0?ooool0gooo000Eool0
0`00ooooo`08ool00`00ooooo`07ool5000>ool00oP0ooooo`3Pool001Goo`03003ooooo00Soo`03
003ooooo01Woo`03n03ooooo0>7oo`005Ooo00<00?ooool01ooo00<00?ooool06Ooo00?h0?ooool0
h_oo000Eool00`00ooooo`07ool00`00ooooo`0Hool00oP0ooooo`3Sool001Goo`8000Ooo`03003o
oooo01Soo`03n03ooooo0>Coo`005Ooo00<00?ooool01_oo00<00?ooool06?oo00?h0?ooool0i?oo
000Eool00`00ooooo`05ool00`00ooooo`0Hool00oP0ooooo`3Uool001Goo`03003ooooo00Goo`03
003ooooo01Ooo`03n03ooooo0>Koo`005Ooo00<00?ooool01Ooo00<00?ooool05_oo00?h0?ooool0
iooo000Eool00`00ooooo`04ool00`00ooooo`0Fool00oP0ooooo`3Xool001Goo`03003ooooo00Co
o`03003ooooo01Coo`;h0>_oo`005Ooo0P001Ooo00<00?ooool04ooo00?h0?ooool0jooo000Eool0
0`00ooooo`03ool00`00ooooo`0Cool00oP0ooooo`3/ool001Goo`03003ooooo00?oo`03003ooooo
01;oo`03n03ooooo0>goo`005Ooo00<00?ooool00_oo00<00?ooool04_oo00?h0?ooool0k_oo000E
ool00`00ooooo`02ool00`00ooooo`0@ool2n03aool000goo`@000Coo`03003ooooo00;oo`03003o
oooo00ooo`03n03ooooo0?7oo`003ooo00<00?ooool00ooo00D00?ooooooo`00013oo`;h0?Coo`00
3ooo00<00?ooool00ooo0P000_oo00<00?ooool03?oo0_P0m_oo000?ool00`00ooooo`03ool01000
ooooo`003Ooo0_P0n?oo000?ool00`00ooooo`03ool01000ooooo`002ooo0_P0n_oo000=ool30005
ool00`00ool0000:ool2n03lool000ooo`03003ooooo00?oo`03003oo`0000Soo`;h0?koo`005Ooo
0`001_oo0_P0oooo0Ooo000Eool30004ool2n03oool3ool001Goo`<000Ch0?ooo`Goo`005Ooo0`00
oooo2Ooo000Eool3003oool9ool001Goo`03003ooooo0?ooo`Woo`005Ooo00?h0?ooool0oooo2Ooo
000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h0000ool0oooo2Ooo
000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h0?ooool0oooo2Ooo
000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h0?ooool0oooo2Ooo
000Eool00oP0003oo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h0?ooool0oooo2Ooo
000Eool00oP0ooooo`3oool9ool001Goo`03n03ooooo0?ooo`Woo`005Ooo00?h0?ooool0oooo2Ooo
00000ooo000000020003ool00`00ooooo`02ool40004ool00oP0ooooo`3oool9ool000?oo`03003o
oooo00Ooo`03003ooooo00Goo`03n03ooooo0?ooo`Woo`000ooo00<00?ooool02?oo00<00?ooool0
1?oo00?h0000ool0oooo2Ooo0003ool00`00ooooo`09ool00`00ooooo`03ool00oP0ooooo`3oool9
ool000?oo`03003ooooo00[oo`03003ooooo00;oo`03n03ooooo0?ooo`Woo`0000Coo`00000000Wo
o`04003ooooo0004ool00oP0ooooo`3oool9ool000?oo`03003ooooo00Soo`8000Goo`03n03ooooo
0?ooo`Woo`005Ooo0`00oooo2Ooo000Eool3003oool9ool001Goo`<00?ooo`Woo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {192.938, 0}} -> {-0.148945, -0.0919099, \
0.00698541, 0.00698541}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Event Direction", "Subsection"],

Cell[TextData[{
  "This example illustrates the solution of the restricted two body problem, \
a standard nonstiff test system of four equations. The example traces the \
path of a spaceship travelling around the moon and returning to the earth \
(see p. 246 of ",
  "[",
  ButtonBox["SG75",
    ButtonData:>{"References.nb", "SG75"},
    ButtonStyle->"Hyperlink"],
  "]",
  "). The ability to specify multiple events and the direction of the zero \
crossing is important."
}], "Text"],

Cell[TextData[{
  "The initial conditions have been chosen to make the orbit periodic. The \
value of ",
  StyleBox["\[Mu]", "MR"],
  " corresponds to a spaceship travelling around the moon and the earth."
}], "MathCaption"],

Cell[BoxData[{\(\[Mu]\  = \ 1/82.45;\), "\[IndentingNewLine]", 
    RowBox[{\(\(\[Mu]\^*\)\  = \ 1\  - \ \[Mu];\), 
      " "}], "\[IndentingNewLine]", 
    RowBox[{" ", \(r\_1\  = \ \@\(\((y\_1[t] + \[Mu])\)\^2 + y\_2[t]\^2\);\), 
      "  "}], "\[IndentingNewLine]", 
    RowBox[{\(r\_2\  = \ \@\(\((y\_1[t] - \(\[Mu]\^*\))\)\^2 + \
y\_2[t]\^2\);\), " "}], "\[IndentingNewLine]", 
    RowBox[{" ", 
      RowBox[{
        RowBox[{"eqns", " ", "=", " ", 
          RowBox[{"(", GridBox[{
                {\(\(y\_1'\)[t]\  \[Equal] \ 
                    y\_3[t]\), \(y\_1[0]\  \[Equal] \ 1.2\)},
                {\(\(y\_2'\)[t]\  \[Equal] \ 
                    y\_4[t]\), \(y\_2[0]\  \[Equal] \ 0\)},
                {\(\(y\_3'\)[t]\  \[Equal] \ 
                    2  y\_4[t]\  + \ 
                      y\_1[t] - \(\(\[Mu]\^*\) \((y\_1[t]\  + \ \[Mu])\)\)\/r\
\_1\^3 - \ \(\[Mu] \((y\_1[t]\  - \ \(\[Mu]\^*\))\)\)\/r\_2\^3\), \(y\_3[
                      0]\  \[Equal] \ 0\)},
                {\(\(y\_4'\)[t]\  \[Equal] \ \(-2\) y\_3[t] + 
                      y\_2[t] - \(\(\[Mu]\^*\) \(y\_2[t]\)\(\ \)\)\/r\_1\^3 - \
\ \(\(\[Mu]\)\(\ \)\(y\_2[t]\)\(\ \)\)\/r\_2\^3\), \(y\_4[
                      0]\  \[Equal] \ \(-1.04935750983031990726\)\)}
                }], ")"}]}], ";"}], "  "}]}], "Input"],

Cell["\<\
The event function is the derivative of the distance from the \
initial conditions. A local maximum or minimum occurs when the value crosses \
zero.\
\>", "MathCaption"],

Cell[BoxData[
    \(\(\(ddist\  = \ 
        2 \((y\_3[t] \((y\_1[t] - 1.2)\) + 
              y\_4[t] y\_2[t])\);\)\(\ \ \)\)\)], "Input"],

Cell[TextData[{
  "There are two events, which for this example are the same. The first event \
(with ",
  StyleBox["Direction", "MR"],
  " ",
  StyleBox["1", "MR"],
  ") corresponds to the point where the distance from the initial point is a \
local  minimum, so that the spaceship returns to its original position. The \
event action is to store the time of the event in the variable ",
  StyleBox["tfinal", "MR"],
  " and to stop the integration. The second event corresponds to a local \
maximum. The event action is to store the time that the spaceship is farthest \
from the starting position in the variable ",
  StyleBox["tfar", "MR"],
  ". "
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol\  = \ 
      First[NDSolve[
          eqns, {y\_1, y\_2, y\_3, 
            y\_4}, {t, \[Infinity]}, \[IndentingNewLine]Method \[Rule] \
{"\<EventLocator\>", \[IndentingNewLine]"\<Event\>" -> {ddist, \ 
                  ddist}, \[IndentingNewLine]"\<Direction\>" \[Rule] {1, \ \
\(-1\)}, \[IndentingNewLine]"\<EventAction\>" \[RuleDelayed] {Throw[
                    tfinal\  = \ t, "\<StopIntegration\>"], 
                  tfar\  = \ t}, \[IndentingNewLine]\ 
              Method \[Rule] "\<ExplicitRungeKutta\>"}]]\)], "Input",
  CellLabel->"In[20]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          
          TagBox[\(InterpolatingFunction[{{0.`, 6.192169260978017`}}, "<>"]\),
            
            False,
            Editable->False], "[", "t", "]"}], ",", 
        RowBox[{
          
          TagBox[\(InterpolatingFunction[{{0.`, 6.192169260978017`}}, "<>"]\),
            
            False,
            Editable->False], "[", "t", "]"}], ",", 
        RowBox[{
          
          TagBox[\(InterpolatingFunction[{{0.`, 6.192169260978017`}}, "<>"]\),
            
            False,
            Editable->False], "[", "t", "]"}], ",", 
        RowBox[{
          
          TagBox[\(InterpolatingFunction[{{0.`, 6.192169260978017`}}, "<>"]\),
            
            False,
            Editable->False], "[", "t", "]"}]}], "}"}]], "Output",
  CellLabel->"Out[20]="]
}, Open  ]],

Cell["\<\
The first two solution components are coordinates of the body of \
infinitesimal mass, so plotting one against the other gives the orbit of the \
body.\
\>", "MathCaption"],

Cell["\<\
This displays one half-orbit when the spaceship is at the furthest \
point from the initial position.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ParametricPlot[{y\_1[t], \ y\_2[t]}\  /. \ sol, \ {t, 0, 
        tfar}]\)], "Input",
  CellLabel->"In[21]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .28506 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.512075 0.386763 0.265192 0.386763 [
[.12531 .25269 -6 -9 ]
[.12531 .25269 6 0 ]
[.31869 .25269 -12 -9 ]
[.31869 .25269 12 0 ]
[.70546 .25269 -9 -9 ]
[.70546 .25269 9 0 ]
[.89884 .25269 -3 -9 ]
[.89884 .25269 3 0 ]
[.49958 .03313 -24 -4.5 ]
[.49958 .03313 0 4.5 ]
[.49958 .07181 -24 -4.5 ]
[.49958 .07181 0 4.5 ]
[.49958 .11049 -24 -4.5 ]
[.49958 .11049 0 4.5 ]
[.49958 .14916 -24 -4.5 ]
[.49958 .14916 0 4.5 ]
[.49958 .18784 -24 -4.5 ]
[.49958 .18784 0 4.5 ]
[.49958 .22652 -24 -4.5 ]
[.49958 .22652 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .28506 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.12531 .26519 m
.12531 .27144 L
s
[(-1)] .12531 .25269 0 1 Mshowa
.31869 .26519 m
.31869 .27144 L
s
[(-0.5)] .31869 .25269 0 1 Mshowa
.70546 .26519 m
.70546 .27144 L
s
[(0.5)] .70546 .25269 0 1 Mshowa
.89884 .26519 m
.89884 .27144 L
s
[(1)] .89884 .25269 0 1 Mshowa
.125 Mabswid
.16399 .26519 m
.16399 .26894 L
s
.20266 .26519 m
.20266 .26894 L
s
.24134 .26519 m
.24134 .26894 L
s
.28002 .26519 m
.28002 .26894 L
s
.35737 .26519 m
.35737 .26894 L
s
.39605 .26519 m
.39605 .26894 L
s
.43472 .26519 m
.43472 .26894 L
s
.4734 .26519 m
.4734 .26894 L
s
.55075 .26519 m
.55075 .26894 L
s
.58943 .26519 m
.58943 .26894 L
s
.6281 .26519 m
.6281 .26894 L
s
.66678 .26519 m
.66678 .26894 L
s
.74413 .26519 m
.74413 .26894 L
s
.78281 .26519 m
.78281 .26894 L
s
.82149 .26519 m
.82149 .26894 L
s
.86016 .26519 m
.86016 .26894 L
s
.08664 .26519 m
.08664 .26894 L
s
.04796 .26519 m
.04796 .26894 L
s
.00928 .26519 m
.00928 .26894 L
s
.93751 .26519 m
.93751 .26894 L
s
.97619 .26519 m
.97619 .26894 L
s
.25 Mabswid
0 .26519 m
1 .26519 L
s
.51208 .03313 m
.51833 .03313 L
s
[(-0.6)] .49958 .03313 1 0 Mshowa
.51208 .07181 m
.51833 .07181 L
s
[(-0.5)] .49958 .07181 1 0 Mshowa
.51208 .11049 m
.51833 .11049 L
s
[(-0.4)] .49958 .11049 1 0 Mshowa
.51208 .14916 m
.51833 .14916 L
s
[(-0.3)] .49958 .14916 1 0 Mshowa
.51208 .18784 m
.51833 .18784 L
s
[(-0.2)] .49958 .18784 1 0 Mshowa
.51208 .22652 m
.51833 .22652 L
s
[(-0.1)] .49958 .22652 1 0 Mshowa
.125 Mabswid
.51208 .04087 m
.51583 .04087 L
s
.51208 .0486 m
.51583 .0486 L
s
.51208 .05634 m
.51583 .05634 L
s
.51208 .06407 m
.51583 .06407 L
s
.51208 .07955 m
.51583 .07955 L
s
.51208 .08728 m
.51583 .08728 L
s
.51208 .09502 m
.51583 .09502 L
s
.51208 .10275 m
.51583 .10275 L
s
.51208 .11822 m
.51583 .11822 L
s
.51208 .12596 m
.51583 .12596 L
s
.51208 .13369 m
.51583 .13369 L
s
.51208 .14143 m
.51583 .14143 L
s
.51208 .1569 m
.51583 .1569 L
s
.51208 .16463 m
.51583 .16463 L
s
.51208 .17237 m
.51583 .17237 L
s
.51208 .1801 m
.51583 .1801 L
s
.51208 .19557 m
.51583 .19557 L
s
.51208 .20331 m
.51583 .20331 L
s
.51208 .21104 m
.51583 .21104 L
s
.51208 .21878 m
.51583 .21878 L
s
.51208 .23425 m
.51583 .23425 L
s
.51208 .24199 m
.51583 .24199 L
s
.51208 .24972 m
.51583 .24972 L
s
.51208 .25746 m
.51583 .25746 L
s
.51208 .0254 m
.51583 .0254 L
s
.51208 .01766 m
.51583 .01766 L
s
.51208 .00993 m
.51583 .00993 L
s
.51208 .00219 m
.51583 .00219 L
s
.51208 .27293 m
.51583 .27293 L
s
.51208 .28066 m
.51583 .28066 L
s
.25 Mabswid
.51208 0 m
.51208 .28506 L
s
0 0 m
1 0 L
1 .28506 L
0 .28506 L
closepath
clip
newpath
.5 Mabswid
.97619 .26519 m
.97619 .26365 L
.97617 .26211 L
.97611 .25903 L
.97586 .25287 L
.97488 .24059 L
.97056 .21436 L
.96395 .19064 L
.95522 .16837 L
.94339 .14558 L
.93032 .12581 L
.91412 .10626 L
.89634 .08917 L
.87825 .07533 L
.85721 .0628 L
.83641 .05363 L
.81513 .04719 L
.79128 .04327 L
.78562 .04284 L
.78278 .04269 L
.77994 .04259 L
.77851 .04256 L
.77709 .04254 L
.77566 .04254 L
.77423 .04254 L
.7728 .04256 L
.77137 .04259 L
.7685 .04268 L
.76539 .04284 L
.76227 .04306 L
.75603 .04368 L
.74351 .04564 L
.71889 .05245 L
.69608 .0626 L
.67167 .07817 L
.6493 .09756 L
.62543 .12486 L
.60171 .15984 L
.57693 .20343 L
.55896 .23464 L
.5458 .25375 L
.53635 .26446 L
.52215 .27507 L
.51255 .27827 L
.50171 .2778 L
.49235 .27363 L
.48532 .26798 L
.47567 .25637 L
.46904 .24561 L
.45277 .21024 L
Mistroke
.4328 .15975 L
.41386 .11913 L
.39572 .08934 L
.37492 .06359 L
.35329 .04375 L
.33219 .02947 L
.30853 .01815 L
.28593 .01121 L
.26354 .00757 L
.25745 .0071 L
.25441 .00694 L
.25137 .00684 L
.24985 .00681 L
.24833 .00679 L
.24681 .00679 L
.24529 .0068 L
.24377 .00682 L
.24225 .00685 L
.23922 .00696 L
.23358 .0073 L
.22795 .00781 L
.21675 .00936 L
.19288 .015 L
.17128 .02296 L
.15094 .03312 L
.13003 .0466 L
.11178 .06132 L
.09357 .07939 L
.07745 .09905 L
.06412 .11892 L
.05166 .1419 L
.042 .16449 L
.03446 .1875 L
.02852 .21319 L
.02512 .23763 L
.02381 .26439 L
.02381 .26519 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[21]:=",
  ImageSize->{288, 82.0625},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000DQ000`40n007h00OT_oo00<00?ooool0
Rooo002Bool3002;ool004Coo`L004Ooo`03003ooooo08_oo`00?Ooo1`001ooo1000@ooo0`00Rooo
000jool3000Bool5000nool00`00ooooo`2;ool003Ooo`<001[oo`@003[oo`<008_oo`00=?oo0`00
8Ooo0P00>?oo00<00?ooool0Rooo000aool3000Vool2000fool00`00ooooo`2;ool002ooo`8002_o
o`80027oo`8000Coo`03003ooooo00?oo`8000Goo`<008_oo`00;Ooo0P00;ooo0P007_oo00@00?oo
ool000Soo`04003ooooo0004ool00`00ooooo`2;ool002coo`03003ooooo037oo`8001coo`04003o
oooo0008ool01000ooooo`001?oo0`00Rooo000Zool2000fool00`00ooooo`0Cool40002ool01000
ooooo`002?oo0`001Ooo00<00?ooool0Rooo000Yool00`00ooooo`0gool2000Iool01000ooooo`00
2?oo00<00?ooool01Ooo0`00Rooo000Wool2000lool00`00ooooo`0Fool01000ooooo`002?oo00<0
0?ooool01Ooo00<00?ooool0?_oo4`00>_oo000Vool00`00ooooo`0mool00`00ooooo`0Fool2000:
ool30004ool3000jool4000Cool5000eool002Goo`03003ooooo03ooo`03003ooooo02Soo`03003o
oooo03Ooo`<001coo`<003;oo`008ooo0P00@ooo00<00?ooool09ooo0`00=Ooo0P008_oo0`00;ooo
000Rool00`00ooooo`14ool00`00ooooo`0Vool00`00ooooo`0cool2000Wool3000/ool0027oo`03
003ooooo04Koo`03003ooooo02Goo`03003ooooo03;oo`03003ooooo02[oo`8002[oo`008?oo00<0
0?ooool0B?oo00<00?ooool04Ooo0P001?oo00<00?ooool00_oo0`001Ooo0`00<?oo0P00;ooo0P00
:?oo000Nool2001<ool00`00ooooo`0?ool01000ooooo`002ooo00<00?ooool00_oo00<00?ooool0
;ooo00<00?ooool0<Ooo00<00?ooool09Ooo000Mool00`00ooooo`1=ool00`00ooooo`0>ool01000
ooooo`002ooo00<00?ooool00_oo0`00;_oo00<00?ooool0<ooo0P009Ooo000Lool00`00ooooo`1?
ool00`00ooooo`07ool40002ool01000ooooo`002?oo0`001Ooo00<00?ooool0;?oo0P00>?oo00<0
0?ooool08_oo000Kool00`00ooooo`1@ool00`00ooooo`0=ool01000ooooo`002?oo00<00?ooool0
1Ooo0`00:ooo00<00?ooool0>Ooo00<00?ooool08Ooo000Jool00`00ooooo`1Bool00`00ooooo`0<
ool01000ooooo`002?oo00<00?ooool01Ooo00<00?ooool0:Ooo0P00?Ooo0P008Ooo000Jool00`00
ooooo`1Cool00`00ooooo`0<ool20009ool40004ool3000Xool00`00ooooo`0oool00`00ooooo`0N
ool001Woo`03003ooooo05Goo`03003ooooo01koo`03003ooooo02Ooo`03003ooooo047oo`03003o
oooo01goo`006?oo00<00?ooool0Eooo00<00?ooool07Ooo0`009Ooo0P00AOoo00<00?ooool07?oo
000Hool00`00ooooo`1Gool00`00ooooo`0Mool00`00ooooo`0Tool00`00ooooo`16ool00`00oooo
o`0Kool001Ooo`03003ooooo05Woo`03003ooooo01coo`03003ooooo02?oo`03003ooooo04Soo`03
003ooooo01[oo`005_oo00<00?ooool0Fooo00<00?ooool02?oo0P001?oo00<00?ooool01?oo00<0
0?ooool00ooo0`008ooo00<00?ooool0BOoo00<00?ooool06Ooo000Eool00`00ooooo`1Mool00`00
ooooo`06ool01000ooooo`002_oo00<00?ooool00ooo00<00?ooool08_oo00<00?ooool0Booo00<0
0?ooool06?oo000Eool00`00ooooo`1Mool00`00ooooo`06ool01000ooooo`001ooo1@001?oo0`00
8Ooo00<00?ooool0COoo00<00?ooool05ooo000Dool00`00ooooo`1Oool00`00ool000030002ool0
1000ooooo`001ooo00@00?ooool000Goo`03003ooooo023oo`03003ooooo04ooo`03003ooooo01Ko
o`004ooo00<00?ooool0HOoo00<00?ooool01?oo00@00?ooool000Soo`03003oo`0000Goo`<0023o
o`03003ooooo04ooo`03003ooooo01Koo`004ooo00<00?ooool0HOoo00<00?ooool01?oo00@00?oo
ool000Woo`8000Goo`03003ooooo01ooo`03003ooooo057oo`03003ooooo01Goo`004_oo00<00?oo
ool0Hooo00<00?ooool01?oo0P002ooo00<00?ooool00ooo0`007_oo00<00?ooool0Dooo00<00?oo
ool05?oo000Bool00`00ooooo`1Sool00`00ooooo`0Gool00`00ooooo`0Mool00`00ooooo`1Eool0
0`00ooooo`0Cool0017oo`03003ooooo06Goo`03003ooooo01Koo`<001goo`03003ooooo05Goo`03
003ooooo01?oo`004Ooo00<00?ooool0IOoo00<00?ooool05_oo00<00?ooool07?oo00<00?ooool0
Eooo00<00?ooool04_oo000@ool00`00ooooo`1Vool00`00ooooo`0Fool00`00ooooo`0Kool00`00
ooooo`1Iool00`00ooooo`0Aool0013oo`03003ooooo06Ooo`03003ooooo00;oo`8000Coo`03003o
oooo00?oo`8000Goo`<001_oo`03003ooooo05Woo`03003ooooo017oo`004?oo00<00?ooool0Iooo
00D00?ooooooo`0000;oo`03003ooooo00Koo`04003ooooo0004ool00`00ooooo`0Jool00`00oooo
o`1Kool00`00ooooo`0@ool000ooo`03003ooooo06Woo`04003ooooo0002ool00`00ooooo`09ool0
0`00ooooo`02ool3000Iool00`00ooooo`1Lool00`00ooooo`0@ool000ooo`03003ooooo06Koo`@0
00;oo`04003ooooo0009ool20005ool00`00ooooo`0Hool00`00ooooo`1Nool00`00ooooo`0?ool0
00koo`03003ooooo06_oo`03003oo`0000;oo`03003ooooo00Woo`03003ooooo00;oo`<001Soo`03
003ooooo05koo`03003ooooo00ooo`003_oo00<00?ooool0Jooo00<00?oo00000_oo00<00?ooool0
1_oo00@00?ooool000Coo`03003ooooo01Ooo`03003ooooo063oo`03003ooooo00koo`003Ooo00<0
0?ooool0K?oo00<00?ooool00P002_oo0P001Ooo0`005_oo00<00?ooool0HOoo00<00?ooool03_oo
000=ool00`00ooooo`1]ool00`00ooooo`0Bool00`00ooooo`0Fool00`00ooooo`1Rool00`00oooo
o`0=ool000coo`03003ooooo06koo`03003ooooo01;oo`<001Goo`03003ooooo06?oo`03003ooooo
00goo`003?oo00<00?ooool0Kooo00<00?ooool04Ooo00<00?ooool05Ooo00<00?ooool0I?oo00<0
0?ooool03?oo000<ool00`00ooooo`1_ool00`00ooooo`0Aool00`00ooooo`0Dool00`00ooooo`1U
ool00`00ooooo`0<ool000coo`03003ooooo073oo`8000Coo`03003ooooo00;oo`@000Coo`<001Co
o`03003ooooo06Koo`03003ooooo00_oo`002ooo00<00?ooool0L?oo0P0000?oo`00ool01ooo00<0
0?ooool01Ooo00<00?ooool04ooo00<00?ooool0Iooo00<00?ooool02ooo000;ool00`00ooooo`1`
ool200000ooo003oo`08ool00`00ooooo`04ool3000Bool00`00ooooo`1Yool00`00ooooo`0:ool0
00_oo`03003ooooo06[oo`@000;oo`04003oo`00000:ool00`00ooooo`03ool00`00ooooo`0Bool0
0`00ooooo`1Yool00`00ooooo`0:ool000[oo`03003ooooo077oo`04003oo`00000;ool00`00oooo
o`02ool3000Aool00`00ooooo`1Zool00`00ooooo`0:ool000[oo`03003ooooo077oo`04003ooooo
0008ool01000ooooo`001?oo00<00?ooool04Ooo00<00?ooool0Jooo00<00?ooool02Ooo000:ool0
0`00ooooo`1bool30009ool20005ool3000@ool00`00ooooo`1/ool00`00ooooo`09ool000[oo`03
003ooooo07Goo`03003ooooo00goo`03003ooooo00ooo`03003ooooo06goo`03003ooooo00Woo`00
2Ooo00<00?ooool0M_oo00<00?ooool03Ooo0`003ooo00<00?ooool0K_oo00<00?ooool02?oo0009
ool00`00ooooo`1fool00`00ooooo`0=ool00`00ooooo`0>ool00`00ooooo`1_ool00`00ooooo`08
ool000Woo`03003ooooo07Ooo`03003ooooo00coo`03003ooooo00koo`03003ooooo06ooo`03003o
oooo00Soo`002Ooo00<00?ooool0Looo0P000_oo00<00?oo00001?oo10001?oo0`003Ooo00<00?oo
ool0L?oo00<00?ooool02?oo0009ool00`00ooooo`1bool01000ooooo`000_oo00<00?ooool01Ooo
00<00?ooool00ooo00<00?ooool03Ooo00<00?ooool0L?oo00<00?ooool02?oo0008ool00`00oooo
o`0Jool4000^ool20004ool00`00ooooo`02ool3000Iool01000ooooo`000_oo00<00?ooool01Ooo
00<00?ooool00ooo0`003?oo00<00?ooool07_oo0P001?oo00<00?ooool00_oo0`00;_oo10004ooo
00<00?ooool02?oo0008ool00`00ooooo`0Lool00`00ooooo`0/ool01000ooooo`002ooo00<00?oo
ool04?oo10000_oo00@00?ooool000;oo`03003ooooo00Goo`03003ooooo00?oo`03003ooooo00co
o`03003ooooo01goo`04003ooooo000;ool00`00ooooo`0]ool00`00ooooo`0Cool00`00ooooo`07
ool000Soo`03003ooooo01coo`03003ooooo02coo`04003ooooo000;ool00`00ooooo`0Fool01000
ooooo`000ooo00<00?ooool01?oo00<00?ooool00ooo0`002ooo00<00?ooool07_oo00@00?ooool0
00_oo`03003ooooo02goo`03003ooooo01?oo`03003ooooo00Ooo`002?oo00<00?ooool05?oo1000
1?oo00<00?ooool09_oo10000_oo00@00?ooool000Soo`<001Woo`04003ooooo0003ool00`00oooo
o`02ool30005ool00`00ooooo`0:ool00`00ooooo`0Oool01000ooooo`002?oo0`00<?oo00<00?oo
ool04ooo00<00?ooool01ooo0008ool00`00ooooo`0Lool00`00ooooo`0/ool01000ooooo`002?oo
00<00?ooool06_oo0P001Ooo00<00?ooool00ooo00<00?ooool00ooo0`002_oo00<00?ooool07ooo
00@00?ooool000Soo`03003ooooo033oo`03003ooooo01?oo`03003ooooo00Ooo`002?oo00<00?oo
ool06_oo0`00;_oo00@00?ooool000Soo`03003ooooo027oo`03003ooooo00Woo`03003ooooo00Wo
o`03003ooooo023oo`04003ooooo0008ool00`00ooooo`0^ool3000Eool00`00ooooo`07ool000So
o`03003ooooo01coo`03003ooooo02goo`8000Woo`@0027oo`03003ooooo00Soo`<000Soo`03003o
oooo02;oo`8000Woo`@002ooo`03003ooooo01?oo`03003ooooo00Ooo`002?oo00<00?ooool0O?oo
00<00?ooool02?oo00<00?ooool02?oo00<00?ooool0M_oo00<00?ooool01ooo0008ool00`00oooo
o`1mool00`00ooooo`07ool00`00ooooo`07ool00`00ooooo`1gool00`00ooooo`07ool000Soo`03
003ooooo07koo`03003ooooo00Koo`<000Koo`03003ooooo07Soo`03003ooooo00Ooo`002?oo00<0
0?ooool0O_oo00<00?ooool01_oo00<00?ooool01Ooo00<00?ooool0NOoo00<00?ooool01ooo0000
0ooo0000003o000L0002ool000?oo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03
003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03
003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`05003oooooool00006
ool00`00ooooo`02ool20004ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00
ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00
ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`07ool002Coo`03003ooooo03Co
o`03003ooooo02koo`8000Coo`<00003ool00?oo037oo`03003ooooo03Coo`03003ooooo01goo`00
S_oo0`0000?oo`00ool00P00R_oo002Aool4002;ool009;oo`03003ooooo08_oo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {81.0625, 0}} -> {-1.33596, -0.685671, \
0.00909225, 0.00909225}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[21]="]
}, Open  ]],

Cell["\<\
This displays one complete orbit when the spaceship returns to the \
initial position.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ParametricPlot[{y\_1[t], \ y\_2[t]}\  /. \ sol, \ {t, 0, 
        tfinal}]\)], "Input",
  CellLabel->"In[22]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .54265 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.512075 0.386763 0.271325 0.386763 [
[.12531 .25882 -6 -9 ]
[.12531 .25882 6 0 ]
[.31869 .25882 -12 -9 ]
[.31869 .25882 12 0 ]
[.70546 .25882 -9 -9 ]
[.70546 .25882 9 0 ]
[.89884 .25882 -3 -9 ]
[.89884 .25882 3 0 ]
[.49957 .03927 -24 -4.5 ]
[.49957 .03927 0 4.5 ]
[.49957 .11662 -24 -4.5 ]
[.49957 .11662 0 4.5 ]
[.49957 .19397 -24 -4.5 ]
[.49957 .19397 0 4.5 ]
[.49957 .34868 -18 -4.5 ]
[.49957 .34868 0 4.5 ]
[.49957 .42603 -18 -4.5 ]
[.49957 .42603 0 4.5 ]
[.49957 .50338 -18 -4.5 ]
[.49957 .50338 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .54265 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.12531 .27132 m
.12531 .27757 L
s
[(-1)] .12531 .25882 0 1 Mshowa
.31869 .27132 m
.31869 .27757 L
s
[(-0.5)] .31869 .25882 0 1 Mshowa
.70546 .27132 m
.70546 .27757 L
s
[(0.5)] .70546 .25882 0 1 Mshowa
.89884 .27132 m
.89884 .27757 L
s
[(1)] .89884 .25882 0 1 Mshowa
.125 Mabswid
.16399 .27132 m
.16399 .27507 L
s
.20266 .27132 m
.20266 .27507 L
s
.24134 .27132 m
.24134 .27507 L
s
.28002 .27132 m
.28002 .27507 L
s
.35737 .27132 m
.35737 .27507 L
s
.39605 .27132 m
.39605 .27507 L
s
.43472 .27132 m
.43472 .27507 L
s
.4734 .27132 m
.4734 .27507 L
s
.55075 .27132 m
.55075 .27507 L
s
.58943 .27132 m
.58943 .27507 L
s
.6281 .27132 m
.6281 .27507 L
s
.66678 .27132 m
.66678 .27507 L
s
.74413 .27132 m
.74413 .27507 L
s
.78281 .27132 m
.78281 .27507 L
s
.82149 .27132 m
.82149 .27507 L
s
.86016 .27132 m
.86016 .27507 L
s
.08664 .27132 m
.08664 .27507 L
s
.04796 .27132 m
.04796 .27507 L
s
.00928 .27132 m
.00928 .27507 L
s
.93751 .27132 m
.93751 .27507 L
s
.97619 .27132 m
.97619 .27507 L
s
.25 Mabswid
0 .27132 m
1 .27132 L
s
.51207 .03927 m
.51832 .03927 L
s
[(-0.6)] .49957 .03927 1 0 Mshowa
.51207 .11662 m
.51832 .11662 L
s
[(-0.4)] .49957 .11662 1 0 Mshowa
.51207 .19397 m
.51832 .19397 L
s
[(-0.2)] .49957 .19397 1 0 Mshowa
.51207 .34868 m
.51832 .34868 L
s
[(0.2)] .49957 .34868 1 0 Mshowa
.51207 .42603 m
.51832 .42603 L
s
[(0.4)] .49957 .42603 1 0 Mshowa
.51207 .50338 m
.51832 .50338 L
s
[(0.6)] .49957 .50338 1 0 Mshowa
.125 Mabswid
.51207 .0586 m
.51582 .0586 L
s
.51207 .07794 m
.51582 .07794 L
s
.51207 .09728 m
.51582 .09728 L
s
.51207 .13596 m
.51582 .13596 L
s
.51207 .1553 m
.51582 .1553 L
s
.51207 .17463 m
.51582 .17463 L
s
.51207 .21331 m
.51582 .21331 L
s
.51207 .23265 m
.51582 .23265 L
s
.51207 .25199 m
.51582 .25199 L
s
.51207 .29066 m
.51582 .29066 L
s
.51207 .31 m
.51582 .31 L
s
.51207 .32934 m
.51582 .32934 L
s
.51207 .36802 m
.51582 .36802 L
s
.51207 .38735 m
.51582 .38735 L
s
.51207 .40669 m
.51582 .40669 L
s
.51207 .44537 m
.51582 .44537 L
s
.51207 .46471 m
.51582 .46471 L
s
.51207 .48404 m
.51582 .48404 L
s
.51207 .01993 m
.51582 .01993 L
s
.51207 .00059 m
.51582 .00059 L
s
.51207 .52272 m
.51582 .52272 L
s
.51207 .54206 m
.51582 .54206 L
s
.25 Mabswid
.51207 0 m
.51207 .54265 L
s
0 0 m
1 0 L
1 .54265 L
0 .54265 L
closepath
clip
newpath
.5 Mabswid
.97619 .27132 m
.97617 .26824 L
.97611 .26516 L
.97586 .259 L
.97488 .24673 L
.971 .2225 L
.95443 .17277 L
.93001 .13152 L
.89894 .09757 L
.85872 .06972 L
.81655 .05367 L
.7924 .04951 L
.78625 .04901 L
.78317 .04884 L
.78008 .04873 L
.77698 .04867 L
.77388 .04868 L
.77077 .04874 L
.76765 .04885 L
.76459 .04903 L
.76152 .04926 L
.75537 .04989 L
.74304 .05186 L
.71836 .05877 L
.67291 .08338 L
.62521 .13128 L
.60288 .16407 L
.59108 .1841 L
.57807 .2075 L
.56236 .23519 L
.55181 .2517 L
.53624 .2707 L
.52352 .28048 L
.50393 .2844 L
.4869 .27558 L
.47684 .26417 L
.46481 .2437 L
.45685 .22624 L
.43412 .1691 L
.41518 .12778 L
.37811 .0732 L
.33338 .03629 L
.28619 .0174 L
.26366 .01371 L
.25802 .01326 L
.2552 .01311 L
.25238 .013 L
.24956 .01294 L
.24674 .01292 L
.24393 .01295 L
Mistroke
.24112 .01302 L
.23807 .01315 L
.23503 .01333 L
.22896 .01384 L
.21689 .01547 L
.19316 .02105 L
.15106 .03919 L
.11017 .06891 L
.07638 .10663 L
.05173 .14789 L
.03344 .19739 L
.0281 .22165 L
.02623 .23395 L
.02489 .24634 L
.02409 .25853 L
.02389 .26464 L
.02383 .2677 L
.02381 .27075 L
.02382 .27381 L
.02386 .27687 L
.02404 .28298 L
.02479 .29518 L
.02818 .32144 L
.03396 .34721 L
.05092 .39309 L
.07782 .43796 L
.11183 .47523 L
.14918 .50239 L
.19432 .52195 L
.21652 .52712 L
.22779 .52869 L
.23915 .52955 L
.24195 .52966 L
.24474 .52971 L
.24754 .52973 L
.25034 .5297 L
.25315 .52962 L
.25595 .52951 L
.26156 .52913 L
.27277 .52783 L
.28396 .52576 L
.33153 .50743 L
.37308 .47486 L
.41359 .41787 L
.43142 .38006 L
.44075 .35697 L
.45143 .32962 L
.46629 .29603 L
.47952 .27494 L
.49133 .26357 L
Mistroke
.5108 .25802 L
.5283 .2652 L
.53942 .27524 L
.55381 .29388 L
.57212 .32442 L
.59976 .37345 L
.62371 .4094 L
.66797 .4555 L
.71754 .48357 L
.74224 .49062 L
.75458 .49266 L
.76687 .49376 L
.76973 .49388 L
.77258 .49395 L
.77543 .49398 L
.77827 .49396 L
.78111 .49389 L
.78394 .49377 L
.78958 .49339 L
.80077 .49209 L
.81182 .49007 L
.85749 .47358 L
.89556 .448 L
.92719 .41485 L
.95367 .3715 L
.96971 .32582 L
.97618 .27293 L
.97619 .27132 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[22]:=",
  ImageSize->{288, 156.25},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000W1000`40n007h00OT_oo0`00Rooo002B
ool00`00ooooo`2;ool009;oo`03003ooooo08_oo`00T_oo00<00?ooool0Rooo0010ool?0013ool0
0`00ooooo`2;ool003_oo`D000ooo`D003koo`03003ooooo08_oo`00=_oo1@006Ooo0P00??oo0`00
Rooo000dool2000Pool2000jool00`00ooooo`2;ool0037oo`<002Coo`8003Soo`03003ooooo08_o
o`00;ooo0P00:Ooo0P008ooo0P001?oo00<00?ooool00ooo0P001Ooo00<00?ooool0Rooo000]ool2
000]ool2000Pool01000ooooo`002?oo00@00?ooool000Coo`03003ooooo08_oo`00:ooo0P00<Ooo
0P007_oo00@00?ooool000Soo`04003ooooo0004ool3002;ool002Woo`8003Goo`03003ooooo01Go
o`@000;oo`04003ooooo0008ool30005ool00`00ooooo`2;ool002Soo`03003ooooo03Koo`8001_o
o`04003ooooo0008ool00`00ooooo`05ool00`00ooooo`2;ool002Ooo`03003ooooo03Woo`03003o
oooo01Soo`04003ooooo0008ool00`00ooooo`05ool00`00ooooo`14ool<000kool002Goo`8003go
o`03003ooooo01Soo`8000[oo`<000Coo`03003ooooo03koo`H000coo`@003Ooo`009?oo00<00?oo
ool0?_oo0P00:ooo00<00?ooool0>_oo10005_oo0`00=?oo000Sool00`00ooooo`11ool00`00oooo
o`0Xool3000hool2000Mool3000aool0027oo`8004Goo`03003ooooo02Ooo`03003ooooo03Koo`80
02;oo`<002koo`008?oo00<00?ooool0A_oo0P009ooo00<00?ooool0=?oo0P009ooo0`00:ooo000O
ool00`00ooooo`19ool00`00ooooo`0Tool00`00ooooo`0bool2000/ool2000Yool001koo`03003o
oooo04_oo`03003ooooo02?oo`03003ooooo033oo`80033oo`8002Ooo`007Ooo00<00?ooool0COoo
00<00?ooool08_oo00<00?ooool0;_oo0P00=?oo00<00?ooool09?oo000Mool00`00ooooo`1=ool0
0`00ooooo`0Rool3000/ool2000gool00`00ooooo`0Sool001coo`03003ooooo04ooo`03003ooooo
027oo`03003ooooo02_oo`03003ooooo03Soo`8002?oo`006ooo00<00?ooool0DOoo00<00?ooool0
8?oo00<00?ooool0:_oo00<00?ooool0>ooo00<00?ooool08?oo000Jool00`00ooooo`1Bool00`00
ooooo`0Pool00`00ooooo`0Yool00`00ooooo`0mool00`00ooooo`0Oool001Woo`03003ooooo05Co
o`03003ooooo01ooo`03003ooooo02Soo`03003ooooo03ooo`8001ooo`006?oo00<00?ooool0E_oo
00<00?ooool07_oo0`009ooo00<00?ooool0@_oo00<00?ooool07?oo000Hool00`00ooooo`1Fool0
0`00ooooo`0Nool00`00ooooo`0Vool00`00ooooo`14ool00`00ooooo`0Kool001Ooo`03003ooooo
05Soo`03003ooooo01goo`03003ooooo02Goo`03003ooooo04Koo`03003ooooo01[oo`005_oo00<0
0?ooool0F_oo00<00?ooool07?oo00<00?ooool09Ooo00<00?ooool0Aooo00<00?ooool06Ooo000E
ool00`00ooooo`1Lool00`00ooooo`08ool20004ool00`00ooooo`04ool00`00ooooo`03ool00`00
ooooo`0Tool00`00ooooo`19ool00`00ooooo`0Hool001Goo`03003ooooo05coo`03003ooooo00Oo
o`04003ooooo000:ool00`00ooooo`03ool00`00ooooo`0Sool00`00ooooo`1:ool00`00ooooo`0H
ool001Coo`03003ooooo05koo`03003ooooo00Koo`04003ooooo0007ool50004ool3000Rool00`00
ooooo`1<ool00`00ooooo`0Gool001Coo`03003ooooo05ooo`03003oo`0000<000;oo`04003ooooo
0007ool01000ooooo`001Ooo00<00?ooool08Ooo00<00?ooool0C_oo00<00?ooool05_oo000Cool0
0`00ooooo`1Pool00`00ooooo`05ool01000ooooo`002?oo00<00?oo00001Ooo00<00?ooool08?oo
00<00?ooool0D?oo00<00?ooool05Ooo000Cool00`00ooooo`1Qool00`00ooooo`04ool01000oooo
o`002Ooo0P001Ooo00<00?ooool07ooo00<00?ooool0D_oo00<00?ooool05?oo000Bool00`00oooo
o`1Rool00`00ooooo`05ool2000;ool00`00ooooo`03ool00`00ooooo`0Nool00`00ooooo`1Dool0
0`00ooooo`0Cool0017oo`03003ooooo06Coo`03003ooooo01Ooo`<001goo`03003ooooo05Koo`03
003ooooo01;oo`004Ooo00<00?ooool0I?oo00<00?ooool05ooo00<00?ooool07?oo00<00?ooool0
Eooo00<00?ooool04_oo000@ool00`00ooooo`1Vool00`00ooooo`0Fool00`00ooooo`0Lool00`00
ooooo`1Hool00`00ooooo`0Aool0013oo`03003ooooo06Koo`03003ooooo01Koo`03003ooooo01_o
o`03003ooooo05Woo`03003ooooo017oo`003ooo00<00?ooool0Iooo00<00?ooool05_oo00<00?oo
ool06_oo00<00?ooool0Fooo00<00?ooool04?oo000?ool00`00ooooo`1Xool00`00ooooo`0Eool0
0`00ooooo`0Iool00`00ooooo`1Lool00`00ooooo`0@ool000koo`03003ooooo06Woo`03003ooooo
01Goo`<001Woo`03003ooooo05goo`03003ooooo00ooo`003_oo00<00?ooool0J_oo00<00?ooool0
5?oo00<00?ooool06?oo00<00?ooool0Gooo00<00?ooool03_oo000>ool00`00ooooo`1Zool00`00
ooooo`0Dool00`00ooooo`0Gool00`00ooooo`1Pool00`00ooooo`0>ool000goo`03003ooooo06co
o`03003ooooo01?oo`03003ooooo01Ooo`03003ooooo067oo`03003ooooo00goo`003Ooo00<00?oo
ool0K?oo00<00?ooool04ooo00<00?ooool05_oo00<00?ooool0H_oo00<00?ooool03Ooo000=ool0
0`00ooooo`1/ool00`00ooooo`0Cool3000Fool00`00ooooo`1Sool00`00ooooo`0<ool000coo`03
003ooooo06koo`03003ooooo01;oo`03003ooooo01Goo`03003ooooo06Coo`03003ooooo00coo`00
3?oo00<00?ooool0K_oo00<00?ooool04_oo00<00?ooool05Ooo00<00?ooool0IOoo00<00?ooool0
2ooo000;ool00`00ooooo`1`ool00`00ooooo`0Aool00`00ooooo`0Dool00`00ooooo`1Vool00`00
ooooo`0;ool000_oo`03003ooooo073oo`<000Coo`03003ooooo00;oo`@000Coo`03003ooooo01?o
o`03003ooooo06Ooo`03003ooooo00_oo`002ooo00<00?ooool0L?oo0P0000?oo`00ool01ooo00<0
0?ooool01Ooo00<00?ooool04ooo00<00?ooool0J?oo00<00?ooool02_oo000:ool00`00ooooo`1a
ool200000ooo003oo`08ool00`00ooooo`04ool3000Bool00`00ooooo`1Yool00`00ooooo`0:ool0
00[oo`03003ooooo06_oo`@000;oo`800003ool00?oo00Woo`03003ooooo00?oo`03003ooooo01;o
o`03003ooooo06Woo`03003ooooo00[oo`002_oo00<00?ooool0LOoo00@00?oo000000_oo`03003o
oooo00;oo`03003ooooo017oo`03003ooooo06_oo`03003ooooo00Woo`002Ooo00<00?ooool0L_oo
00@00?oo000000Soo`04003ooooo0004ool00`00ooooo`0Aool00`00ooooo`1[ool00`00ooooo`09
ool000Woo`03003ooooo07?oo`<000Woo`8000Goo`03003ooooo013oo`03003ooooo06goo`03003o
oooo00Soo`002Ooo00<00?ooool0MOoo00<00?ooool03_oo0`004?oo00<00?ooool0KOoo00<00?oo
ool02?oo0009ool00`00ooooo`1fool00`00ooooo`0=ool00`00ooooo`0?ool00`00ooooo`1^ool0
0`00ooooo`08ool000Soo`03003ooooo07Ooo`03003ooooo00goo`03003ooooo00ooo`03003ooooo
06ooo`03003ooooo00Ooo`002?oo00<00?ooool0N?oo00<00?ooool03?oo00<00?ooool03_oo00<0
0?ooool0L?oo00<00?ooool01ooo0008ool00`00ooooo`1hool00`00ooooo`0<ool00`00ooooo`0>
ool00`00ooooo`1`ool00`00ooooo`07ool000Ooo`03003ooooo07Woo`03003ooooo00coo`03003o
oooo00goo`03003ooooo077oo`03003ooooo00Ooo`001ooo00<00?ooool06ooo1000;_oo0P001?oo
00<00?ooool00_oo0`007ooo00<00?ooool02ooo0`003Ooo00<00?ooool07_oo0P001?oo00<00?oo
ool00_oo0`00;_oo10004ooo00<00?ooool01ooo0007ool00`00ooooo`0Mool00`00ooooo`0/ool0
1000ooooo`002ooo00<00?ooool07?oo00<00?ooool02ooo00<00?ooool03?oo00<00?ooool07_oo
00@00?ooool000_oo`03003ooooo02goo`03003ooooo01?oo`03003ooooo00Koo`001ooo00<00?oo
ool07Ooo00<00?ooool0;?oo00@00?ooool000_oo`03003ooooo01goo`03003ooooo00[oo`03003o
oooo00_oo`03003ooooo01ooo`04003ooooo000;ool00`00ooooo`0]ool00`00ooooo`0Cool00`00
ooooo`06ool000Ooo`03003ooooo01Goo`@000Coo`03003ooooo02Koo`@000;oo`04003ooooo0008
ool3000Pool00`00ooooo`0:ool00`00ooooo`0;ool00`00ooooo`0Oool01000ooooo`002?oo0`00
<?oo00<00?ooool04ooo00<00?ooool01_oo0007ool00`00ooooo`0Mool00`00ooooo`0/ool01000
ooooo`002?oo00<00?ooool08?oo00<00?ooool02_oo00<00?ooool02_oo00<00?ooool08?oo00@0
0?ooool000Soo`03003ooooo033oo`03003ooooo01?oo`03003ooooo00Koo`001ooo00<00?ooool0
6ooo0`00;_oo00@00?ooool000Soo`03003ooooo027oo`03003ooooo00Woo`<000Woo`03003ooooo
027oo`04003ooooo0008ool00`00ooooo`0^ool3000Eool00`00ooooo`06ool000Ooo`03003ooooo
01goo`03003ooooo02goo`8000Woo`@0023oo`03003ooooo00Woo`03003ooooo00Soo`03003ooooo
02?oo`8000Woo`@002ooo`03003ooooo01?oo`03003ooooo00Koo`001ooo00<00?ooool0OOoo00<0
0?ooool01ooo0`002?oo00<00?ooool0N?oo00<00?ooool01_oo0007ool00`00ooooo`1mool00`00
ooooo`05ool200000ooo003oo`020006ool00`00ooooo`1hool00`00ooooo`06ool000Ooo`03003o
oooo07koo`03003ooooo00;oo`8000?oo`04003oooooool20003ool00`00ooooo`1iool00`00oooo
o`06ool000Ooo`03003ooooo07ooo`04003oo`000005ool00`00ooooo`03ool00`00ool0001lool0
0`00ooooo`06ool00?l00200007oo`000ooo00D00?ooooooo`0000Koo`03003ooooo00Soo`03003o
oooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003o
oooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`03003ooooo00Soo`05003o
o`00ool00006ool00`00ooooo`02ool200000ooo003oo`02ool00`00ooooo`08ool00`00ooooo`08
ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08
ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`06
ool000Ooo`03003ooooo01[oo`03003ooooo03Coo`03003ooooo02[oo`04003oooooool20004ool0
0`00ooooo`020004ool00`00ooooo`0/ool00`00ooooo`0dool00`00ooooo`0Cool00`00ooooo`06
ool000Ooo`03003ooooo07koo`03003ooooo00?oo`8000;oo`<000Ooo`03003ooooo07Soo`03003o
oooo00Koo`001ooo00<00?ooool0OOoo00<00?ooool01_oo0`002Ooo00<00?ooool0Mooo00<00?oo
ool01ooo0007ool00`00ooooo`1lool00`00ooooo`09ool30008ool00`00ooooo`1fool00`00oooo
o`07ool000Ooo`03003ooooo07coo`03003ooooo00Woo`03003ooooo00Woo`03003ooooo07Goo`03
003ooooo00Ooo`001ooo00<00?ooool0Nooo00<00?ooool02_oo00<00?ooool02_oo00<00?ooool0
M?oo00<00?ooool01ooo0007ool00`00ooooo`1kool00`00ooooo`0:ool00`00ooooo`0:ool00`00
ooooo`1dool00`00ooooo`07ool000Ooo`03003ooooo07[oo`03003ooooo00_oo`03003ooooo00_o
o`03003ooooo07?oo`03003ooooo00Ooo`001ooo00<00?ooool0N_oo00<00?ooool02ooo00<00?oo
ool03?oo00<00?ooool0L_oo00<00?ooool01ooo0008ool00`00ooooo`1hool00`00ooooo`0<ool3
000<ool00`00ooooo`1bool00`00ooooo`07ool000Soo`03003ooooo07Soo`03003ooooo00coo`03
003ooooo00goo`03003ooooo073oo`03003ooooo00Soo`002?oo00<00?ooool0N?oo00<00?ooool0
3?oo00<00?ooool03_oo00<00?ooool0Kooo00<00?ooool02?oo0008ool00`00ooooo`1gool00`00
ooooo`0=ool00`00ooooo`0>ool00`00ooooo`1_ool00`00ooooo`08ool000Soo`03003ooooo07Oo
o`03003ooooo00goo`03003ooooo00ooo`03003ooooo06koo`03003ooooo00Soo`002?oo00<00?oo
ool0M_oo00<00?ooool03_oo0`004?oo00<00?ooool0KOoo00<00?ooool02?oo0009ool00`00oooo
o`1eool00`00ooooo`0>ool00`00ooooo`0@ool00`00ooooo`1/ool00`00ooooo`09ool000Woo`03
003ooooo07Goo`03003ooooo00koo`03003ooooo017oo`03003ooooo06_oo`03003ooooo00Woo`00
2Ooo00<00?ooool0M?oo00<00?ooool03ooo00<00?ooool04Ooo00<00?ooool0Jooo00<00?ooool0
2Ooo0009ool00`00ooooo`1cool20004ool00`00ooooo`02ool40004ool00`00ooooo`0Bool00`00
ooooo`1Zool00`00ooooo`09ool000[oo`03003ooooo077oo`800003ool00?oo00Ooo`03003ooooo
00Goo`03003ooooo01;oo`03003ooooo06Woo`03003ooooo00[oo`002_oo00<00?ooool0LOoo0P00
00?oo`00ool02?oo00<00?ooool01?oo0`004ooo00<00?ooool0J?oo00<00?ooool02_oo000:ool0
0`00ooooo`1aool01000ooooo`002_oo00<00?ooool00ooo00<00?ooool05?oo00<00?ooool0Iooo
00<00?ooool02_oo000;ool00`00ooooo`1`ool01000ooooo`002ooo00<00?ooool00_oo00<00?oo
ool05?oo00<00?ooool0I_oo00<00?ooool02ooo000;ool00`00ooooo`1`ool01000ooooo`002?oo
00@00?ooool000Coo`03003ooooo01Goo`03003ooooo06Goo`03003ooooo00_oo`003?oo00<00?oo
ool0K_oo00@00?oo000000[oo`8000Goo`03003ooooo01Goo`03003ooooo06Goo`03003ooooo00_o
o`003?oo00<00?ooool0K_oo00<00?ooool04_oo00<00?ooool05_oo00<00?ooool0I?oo00<00?oo
ool02ooo000<ool00`00ooooo`1]ool00`00ooooo`0Cool3000Fool00`00ooooo`1Sool00`00oooo
o`0<ool000goo`03003ooooo06coo`03003ooooo01?oo`03003ooooo01Ooo`03003ooooo06;oo`03
003ooooo00coo`003Ooo00<00?ooool0Jooo00<00?ooool05?oo00<00?ooool06?oo00<00?ooool0
H?oo00<00?ooool03Ooo000=ool00`00ooooo`1[ool00`00ooooo`0Dool00`00ooooo`0Hool00`00
ooooo`1Pool00`00ooooo`0=ool000koo`03003ooooo06[oo`03003ooooo01Coo`03003ooooo01Wo
o`03003ooooo05koo`03003ooooo00koo`003_oo00<00?ooool0JOoo00<00?ooool05Ooo0`006Ooo
00<00?ooool0GOoo00<00?ooool03ooo000?ool00`00ooooo`1Xool00`00ooooo`0Eool00`00oooo
o`0Jool00`00ooooo`1Lool00`00ooooo`0?ool000ooo`03003ooooo06Ooo`03003ooooo01Koo`03
003ooooo01_oo`03003ooooo05[oo`03003ooooo013oo`004?oo00<00?ooool0I_oo00<00?ooool0
5_oo00<00?ooool06ooo00<00?ooool0FOoo00<00?ooool04Ooo000@ool00`00ooooo`1Uool00`00
ooooo`0Gool00`00ooooo`0Lool00`00ooooo`1Hool00`00ooooo`0Aool0017oo`03003ooooo06Co
o`03003ooooo01Ooo`03003ooooo01coo`03003ooooo05Ooo`03003ooooo01;oo`004Ooo00<00?oo
ool0Hooo00<00?ooool06?oo0`007Ooo00<00?ooool0EOoo00<00?ooool04ooo000Bool00`00oooo
o`1Rool00`00ooooo`0Hool00`00ooooo`0Nool00`00ooooo`1Dool00`00ooooo`0Cool001;oo`03
003ooooo067oo`03003ooooo01Woo`03003ooooo01ooo`03003ooooo05;oo`03003ooooo01Coo`00
4ooo00<00?ooool0H?oo00<00?ooool01_oo0P001?oo00<00?ooool01?oo00<00?ooool00ooo00<0
0?ooool08?oo00<00?ooool0D?oo00<00?ooool05Ooo000Cool00`00ooooo`1Oool00`00ooooo`06
ool01000ooooo`002_oo00<00?ooool00ooo00<00?ooool08Ooo00<00?ooool0C_oo00<00?ooool0
5_oo000Dool00`00ooooo`1Nool00`00ooooo`06ool01000ooooo`001ooo1@001?oo0`008_oo00<0
0?ooool0C?oo00<00?ooool05ooo000Dool00`00ooooo`1Mool00`00ooooo`07ool01000ooooo`00
1ooo00@00?ooool000Goo`03003ooooo02?oo`03003ooooo04[oo`03003ooooo01Soo`005Ooo00<0
0?ooool0Fooo00<00?ooool02?oo00@00?ooool000Soo`03003oo`0000Goo`03003ooooo02?oo`03
003ooooo04Woo`03003ooooo01Woo`005Ooo00<00?ooool0Fooo00<00?ooool02?oo00@00?ooool0
00Woo`8000Goo`03003ooooo02Coo`03003ooooo04Ooo`03003ooooo01[oo`005_oo00<00?ooool0
FOoo00<00?ooool02_oo0P002ooo00<00?ooool00ooo00<00?ooool09Ooo00<00?ooool0AOoo00<0
0?ooool06ooo000Gool00`00ooooo`1Gool00`00ooooo`0Nool00`00ooooo`0Vool00`00ooooo`13
ool00`00ooooo`0Lool001Soo`03003ooooo05Koo`03003ooooo01koo`<002Ooo`03003ooooo047o
o`03003ooooo01goo`006Ooo00<00?ooool0E?oo00<00?ooool07ooo00<00?ooool0:?oo00<00?oo
ool0?_oo0P008?oo000Jool00`00ooooo`1Bool00`00ooooo`0Pool00`00ooooo`0Yool00`00oooo
o`0lool00`00ooooo`0Pool001_oo`03003ooooo053oo`03003ooooo027oo`03003ooooo02[oo`03
003ooooo03[oo`03003ooooo027oo`007?oo00<00?ooool0Cooo00<00?ooool08Ooo00<00?ooool0
:ooo0P00>?oo0P009?oo000Mool00`00ooooo`1=ool00`00ooooo`0Rool3000]ool2000eool00`00
ooooo`0Tool001koo`03003ooooo04_oo`03003ooooo02?oo`03003ooooo02ooo`8003;oo`03003o
oooo02Goo`007ooo00<00?ooool0B_oo00<00?ooool08ooo00<00?ooool0<Ooo00<00?ooool0;Ooo
0P00:?oo000Pool00`00ooooo`18ool00`00ooooo`0Tool00`00ooooo`0bool2000[ool2000Zool0
027oo`8004Koo`8002Ooo`03003ooooo03Coo`8002Koo`<002coo`008ooo00<00?ooool0@_oo00<0
0?ooool09ooo00<00?ooool0=_oo0P008?oo1000;ooo000Tool00`00ooooo`10ool00`00ooooo`0X
ool3000hool2000Kool3000cool002Goo`8003koo`8002_oo`03003ooooo03[oo`@001Coo`<003Ko
o`009ooo00<00?ooool0>_oo00<00?ooool0:ooo00<00?ooool0?_oo1`001ooo1P00>Ooo000Xool0
0`00ooooo`0hool00`00ooooo`0Iool20004ool00`00ooooo`03ool20005ool00`00ooooo`15ool7
000oool002Woo`8003Koo`8001_oo`04003ooooo0008ool01000ooooo`001?oo00<00?ooool0Rooo
000[ool2000cool00`00ooooo`0Kool01000ooooo`002?oo00@00?ooool000Coo`<008_oo`00;Ooo
0P00;ooo0P007_oo00@00?ooool000Soo`<000Goo`03003ooooo08_oo`00;ooo0P00:ooo0P008?oo
00@00?ooool000Soo`03003ooooo00Goo`03003ooooo08_oo`00<Ooo0P009ooo0P008_oo00@00?oo
ool000Soo`03003ooooo00Goo`03003ooooo08_oo`00<ooo0P008_oo0`009Ooo0P002_oo0`001?oo
00<00?ooool0Rooo000eool2000Nool2000kool00`00ooooo`2;ool003Ooo`@001Soo`8003goo`<0
08_oo`00>ooo1@003Ooo1P00?ooo00<00?ooool0Rooo0010ool=0015ool00`00ooooo`2;ool009;o
o`03003ooooo08_oo`00T_oo00<00?ooool0Rooo002Bool3002;ool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {155.25, 0}} -> {-1.32809, -0.701528, \
0.0090374, 0.0090374}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[22]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Discontinuous Equations and Switching Functions", "Subsection"],

Cell["\<\
In many applications the function in a differential system may not \
be analytic or continuous everywhere.\
\>", "Text"],

Cell[TextData[{
  "A common discontinuous problem that arises in practice involves a ",
  StyleBox["switching function",
    FontSlant->"Italic"],
  " ",
  Cell[BoxData[
      \(TraditionalForm\`g\)]],
  ":"
}], "Text"],

Cell[BoxData[
    RowBox[{\(y\^'\), "=", 
      RowBox[{"{", GridBox[{
            {\(\(f\_I\) \((t, y)\)\), "if", \(g \((t, y)\)\  > \ 0\)},
            {\(\(f\_II\) \((t, y)\)\), "if", \(g \((t, y)\)\  < \ 0\)}
            }]}]}]], "DisplayMath"],

Cell[TextData[{
  "In order to illustrate the difficulty in crossing a discontinuity, \
consider the following example [",
  ButtonBox["G\[CapitalOSlash]84",
    ButtonData:>{"References.nb", "G\[CapitalOSlash]84"},
    ButtonStyle->"Hyperlink"],
  "] (see also [",
  ButtonBox["HNW93",
    ButtonData:>{"References.nb", "HNW93"},
    ButtonStyle->"Hyperlink"],
  "]):"
}], "Text"],

Cell[BoxData[
    RowBox[{\(y\^'\), "=", 
      RowBox[{"{", GridBox[{
            {\(t\^2 + \ 2\ y\^2\), 
              "if", \(\((t + 1\/20)\)\^2\  + \((y + 3\/20)\)\^2 \[LessEqual] \
\ 1\)},
            {\(\(2  t^2\)\(+\)\(3  y[t]^2\)\(-\)\(2\)\(\ \)\), 
              "if", \(\((t + 1\/20)\)\^2\  + \((y + 3\/20)\)\^2\  > \ 1\)}
            }]}]}]], "DisplayMath"],

Cell[TextData[{
  "Here is the input for the entire system. The switching function is \
assigned to the symbol ",
  StyleBox["event", "MR"],
  " and the function defining the system depends on the sign of the switching \
function."
}], "MathCaption"],

Cell[BoxData[{
    \(\(t0\  = \ 0;\)\), "\[IndentingNewLine]", 
    \(\(ics0\  = \ 3/10;\)\), "\[IndentingNewLine]", 
    \(\(event\  = \ \((\((t + 1/20)\)^2 + \((y[t] + 3/20)\)^2 - 
            1)\);\)\), "\[IndentingNewLine]", 
    \(\(system\  = \ {\(y'\)[t] \[Equal] 
            If[event <= 0, t^2 + 2  y[t]^2, 2  t^2 + 3  y[t]^2 - 2], \ 
          y[t0] \[Equal] ics0};\)\)}], "Input",
  CellLabel->"In[17]:="],

Cell[TextData[{
  "The symbol ",
  StyleBox["odemethod", "MR"],
  " is used to indicate the numerical method that should be used for the \
integration. For comparison, you might want to define a different method, \
such as ",
  StyleBox["ExplicitRungeKutta", "MR"],
  ", and rerun the computations in this section to see how other methods \
behave."
}], "MathCaption"],

Cell[BoxData[
    \(\(odemethod\  = \ Automatic;\)\)], "Input"],

Cell[TextData[{
  "This solves the system on the interval [0,1] and collects data for the \
mesh points of the integration using ",
  StyleBox["Reap", "MR"],
  " and ",
  StyleBox["Sow", "MR"],
  "."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(data\  = \ \(Reap[\[IndentingNewLine]sol\  = 
              y[t]\  /. \ First[
                  NDSolve[system, y, {t, t0, 1}, Method \[Rule] odemethod, \ 
                    MaxStepFraction \[Rule] 1, 
                    StepMonitor \[RuleDelayed] 
                      Sow[t]]]\[IndentingNewLine]]\)[\([2, 
            1]\)];\)\), "\[IndentingNewLine]", 
    \(sol\)}], "Input",
  CellLabel->"In[21]:="],

Cell[BoxData[
    RowBox[{
      TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
        False,
        Editable->False], "[", "t", "]"}]], "Output",
  CellLabel->"Out[22]="]
}, Open  ]],

Cell["Here is a plot of the solution.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(dirsol\  = \ Plot[sol, {t, t0, 1}]\)], "Input",
  CellLabel->"In[23]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 -0.34178 1.18832 [
[.21429 .00222 -9 -9 ]
[.21429 .00222 9 0 ]
[.40476 .00222 -9 -9 ]
[.40476 .00222 9 0 ]
[.59524 .00222 -9 -9 ]
[.59524 .00222 9 0 ]
[.78571 .00222 -9 -9 ]
[.78571 .00222 9 0 ]
[.97619 .00222 -3 -9 ]
[.97619 .00222 3 0 ]
[.01131 .13355 -18 -4.5 ]
[.01131 .13355 0 4.5 ]
[.01131 .25238 -18 -4.5 ]
[.01131 .25238 0 4.5 ]
[.01131 .37121 -18 -4.5 ]
[.01131 .37121 0 4.5 ]
[.01131 .49004 -18 -4.5 ]
[.01131 .49004 0 4.5 ]
[.01131 .60887 -18 -4.5 ]
[.01131 .60887 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .01472 m
.21429 .02097 L
s
[(0.2)] .21429 .00222 0 1 Mshowa
.40476 .01472 m
.40476 .02097 L
s
[(0.4)] .40476 .00222 0 1 Mshowa
.59524 .01472 m
.59524 .02097 L
s
[(0.6)] .59524 .00222 0 1 Mshowa
.78571 .01472 m
.78571 .02097 L
s
[(0.8)] .78571 .00222 0 1 Mshowa
.97619 .01472 m
.97619 .02097 L
s
[(1)] .97619 .00222 0 1 Mshowa
.125 Mabswid
.07143 .01472 m
.07143 .01847 L
s
.11905 .01472 m
.11905 .01847 L
s
.16667 .01472 m
.16667 .01847 L
s
.2619 .01472 m
.2619 .01847 L
s
.30952 .01472 m
.30952 .01847 L
s
.35714 .01472 m
.35714 .01847 L
s
.45238 .01472 m
.45238 .01847 L
s
.5 .01472 m
.5 .01847 L
s
.54762 .01472 m
.54762 .01847 L
s
.64286 .01472 m
.64286 .01847 L
s
.69048 .01472 m
.69048 .01847 L
s
.7381 .01472 m
.7381 .01847 L
s
.83333 .01472 m
.83333 .01847 L
s
.88095 .01472 m
.88095 .01847 L
s
.92857 .01472 m
.92857 .01847 L
s
.25 Mabswid
0 .01472 m
1 .01472 L
s
.02381 .13355 m
.03006 .13355 L
s
[(0.4)] .01131 .13355 1 0 Mshowa
.02381 .25238 m
.03006 .25238 L
s
[(0.5)] .01131 .25238 1 0 Mshowa
.02381 .37121 m
.03006 .37121 L
s
[(0.6)] .01131 .37121 1 0 Mshowa
.02381 .49004 m
.03006 .49004 L
s
[(0.7)] .01131 .49004 1 0 Mshowa
.02381 .60887 m
.03006 .60887 L
s
[(0.8)] .01131 .60887 1 0 Mshowa
.125 Mabswid
.02381 .03848 m
.02756 .03848 L
s
.02381 .06225 m
.02756 .06225 L
s
.02381 .08601 m
.02756 .08601 L
s
.02381 .10978 m
.02756 .10978 L
s
.02381 .15731 m
.02756 .15731 L
s
.02381 .18108 m
.02756 .18108 L
s
.02381 .20485 m
.02756 .20485 L
s
.02381 .22861 m
.02756 .22861 L
s
.02381 .27615 m
.02756 .27615 L
s
.02381 .29991 m
.02756 .29991 L
s
.02381 .32368 m
.02756 .32368 L
s
.02381 .34744 m
.02756 .34744 L
s
.02381 .39498 m
.02756 .39498 L
s
.02381 .41874 m
.02756 .41874 L
s
.02381 .44251 m
.02756 .44251 L
s
.02381 .46628 m
.02756 .46628 L
s
.02381 .51381 m
.02756 .51381 L
s
.02381 .53758 m
.02756 .53758 L
s
.02381 .56134 m
.02756 .56134 L
s
.02381 .58511 m
.02756 .58511 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .01472 m
.02498 .01498 L
.02615 .01524 L
.02848 .01577 L
.03316 .01683 L
.0425 .01897 L
.06277 .02371 L
.08169 .0283 L
.10025 .03296 L
.12037 .03823 L
.13915 .04338 L
.15951 .04924 L
.17949 .05531 L
.19813 .06128 L
.21834 .06813 L
.23721 .0749 L
.25571 .08193 L
.27578 .09004 L
.29451 .09809 L
.31481 .10738 L
.33473 .11712 L
.35332 .1268 L
.37348 .13801 L
.39229 .14918 L
.41268 .16212 L
.4327 .17572 L
.45137 .18928 L
.47162 .20501 L
.49052 .22072 L
.50905 .23716 L
.52915 .25625 L
.54791 .27532 L
.56825 .29748 L
.58821 .32087 L
.59752 .33236 L
.60217 .33826 L
.60683 .34426 L
.61188 .35087 L
.6144 .35423 L
.61566 .35592 L
.61692 .35762 L
.61819 .35831 L
.61945 .35802 L
.62071 .35775 L
.62197 .35747 L
.6245 .35694 L
.62702 .35642 L
.62938 .35595 L
.63173 .3555 L
.63644 .35463 L
Mistroke
.64587 .35308 L
.65511 .35181 L
.65973 .35127 L
.66435 .3508 L
.66936 .35037 L
.67186 .35018 L
.67437 .35001 L
.67688 .34986 L
.67938 .34974 L
.68189 .34964 L
.6844 .34956 L
.68557 .34953 L
.68673 .3495 L
.6879 .34948 L
.68907 .34946 L
.69024 .34945 L
.69141 .34945 L
.69258 .34945 L
.69375 .34945 L
.69492 .34946 L
.69609 .34948 L
.69726 .34949 L
.69843 .34952 L
.6996 .34955 L
.70076 .34959 L
.7031 .34967 L
.70437 .34973 L
.70564 .34979 L
.70817 .34993 L
.71071 .3501 L
.71324 .35029 L
.71831 .35076 L
.72338 .35134 L
.73285 .35271 L
.74232 .35449 L
.7516 .35665 L
.76088 .35923 L
.78102 .3664 L
.79981 .37517 L
.82018 .38718 L
.84018 .40177 L
.85883 .41814 L
.87905 .43921 L
.89794 .46236 L
.91645 .48866 L
.93653 .52173 L
.95527 .55735 L
.97558 .60188 L
.97619 .60332 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[23]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool004?oo`8000Coo`03003ooooo00;oo`@002Coo`8000Coo`03003ooooo00Coo`03003ooooo02?o
o`8000Coo`03003ooooo00?oo`8002Goo`8000Coo`03003ooooo00?oo`8002_oo`@000Ooo`00@_oo
00@00?ooool000Soo`03003ooooo02Coo`04003ooooo000:ool00`00ooooo`0Rool01000ooooo`00
2?oo00@00?ooool002?oo`04003ooooo0008ool01000ooooo`00;?oo00<00?ooool01_oo0012ool0
1000ooooo`002Ooo00<00?ooool08ooo00@00?ooool000Ooo`D002?oo`04003ooooo0008ool01000
ooooo`008ooo00@00?ooool000Soo`04003ooooo000/ool00`00ooooo`06ool004;oo`04003ooooo
000:ool00`00ooooo`0Rool01000ooooo`001ooo00@00?ooool002Coo`04003ooooo0008ool3000T
ool01000ooooo`002Ooo0P00;Ooo00<00?ooool01_oo0012ool01000ooooo`002ooo00<00?ooool0
8Ooo00@00?ooool000Soo`03003oo`0002Coo`04003ooooo0008ool00`00ooooo`0Tool01000oooo
o`002?oo00@00?ooool002coo`03003ooooo00Koo`00@_oo00@00?ooool000Soo`04003ooooo000S
ool01000ooooo`002Ooo0P009?oo00@00?ooool000Soo`03003ooooo02Coo`04003ooooo0008ool0
1000ooooo`00:_oo0`002?oo000Fool00`00ooooo`0Zool2000:ool2000Uool2000;ool00`00oooo
o`0Sool2000:ool3000Tool2000:ool2000]ool00`00ooooo`06ool001Koo`03003ooooo0?ooo`So
o`005_oo00<00?ooool0oooo2?oo000Fool00`00ooooo`3oool8ool0013oool000l000;oo`005_oo
00@00?ooooooo`@000Goo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00Woo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00Woo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00[oo`03003ooooo00Woo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00[oo`03003ooooo00Koo`005_oo00<00?ooool01Ooo1@009ooo00<00?ooool0<?oo00<00?ooool0
<?oo00<00?ooool0<?oo00<00?ooool0<Ooo00<00?ooool01_oo000Fool00`00ooooo`0:ool2003k
ool001Koo`03003ooooo00coo`@00?Ooo`005_oo00<00?ooool04?oo1@00l_oo000Fool2000Fool4
003^ool001Koo`03003ooooo01Woo`800>coo`005_oo00<00?ooool06ooo1000j?oo000Fool00`00
ooooo`0Oool5003Sool001Koo`03003ooooo02Coo`800>7oo`005_oo00<00?ooool09_oo1000gOoo
000Fool00`00ooooo`0Zool4003Iool001Koo`8002ooo`<00=Koo`005_oo00<00?ooool0<Ooo0`00
dooo000Fool00`00ooooo`0dool2003Aool001Koo`03003ooooo03Koo`<00<koo`005_oo00<00?oo
ool0>Ooo0P00c?oo000Fool00`00ooooo`0kool30039ool001Koo`8003ooo`800<Ooo`005_oo00<0
0?ooool0@?oo0`00a?oo000Fool00`00ooooo`13ool20032ool001Koo`03003ooooo04Goo`800<3o
o`005_oo00<00?ooool0Aooo0P00__oo000Fool00`00ooooo`19ool2002lool001Koo`03003ooooo
04_oo`800;[oo`005_oo0P00C_oo0P00^?oo000Fool00`00ooooo`1?ool2002fool001Koo`03003o
oooo057oo`800;Coo`005_oo00<00?ooool0Dooo0P00/_oo0003ool20004ool00`00ooooo`04ool0
0`00ooooo`03ool00`00ooooo`1Eool2002`ool000;oo`04003ooooo000:ool00`00ooooo`03ool0
0`00ooooo`1Gool2002^ool000;oo`04003ooooo0007ool50004ool3001Iool2002/ool000;oo`04
003ooooo0007ool01000ooooo`001Ooo00<00?ooool0Fooo00<00?ooool0ZOoo0002ool01000oooo
o`002?oo00<00?oo00001Ooo00<00?ooool0G?oo0P00ZOoo0002ool01000ooooo`002Ooo0P001Ooo
00<00?ooool0G_oo0P00Yooo0003ool2000;ool00`00ooooo`03ool00`00ooooo`1Pool00`00oooo
o`2Tool001Koo`03003ooooo067oo`800:Coo`005_oo0P00I?oo00<00?ooool0XOoo000Fool00`00
ooooo`1Tool2002Qool001Koo`03003ooooo06Koo`03003ooooo09koo`005_oo00<00?ooool0Iooo
0P00W_oo000Fool00`00ooooo`1Yool2002Lool001Koo`03003ooooo06_oo`03003ooooo09Woo`00
5_oo00<00?ooool0K?oo00<00?ooool0V?oo000Fool2001^ool2002Hool001Koo`03003ooooo06oo
o`03003ooooo09Goo`005_oo00<00?ooool0L?oo00<00?ooool0U?oo000Fool00`00ooooo`1aool2
002Dool001Koo`03003ooooo07?oo`03003ooooo097oo`005_oo00<00?ooool0M?oo0P00TOoo000F
ool2001gool00`00ooooo`2>ool001Koo`03003ooooo07Ooo`03003ooooo08goo`005_oo00<00?oo
ool0N?oo0P00SOoo000Fool00`00ooooo`1jool00`00ooooo`2:ool001Koo`03003ooooo07_oo`03
003ooooo08Woo`005_oo00<00?ooool0O?oo00<00?ooool0R?oo000Fool00`00ooooo`1mool00`00
ooooo`27ool001Koo`8007ooo`03003ooooo08Koo`005_oo00<00?ooool0Oooo00<00?ooool0QOoo
000Fool00`00ooooo`20ool00`00ooooo`24ool001Koo`03003ooooo087oo`03003ooooo08?oo`00
0ooo0P001?oo00<00?ooool00_oo0`001Ooo00<00?ooool0P_oo00<00?ooool0P_oo0002ool01000
ooooo`002ooo00<00?ooool00_oo00<00?ooool0Pooo00<00?ooool0POoo0002ool01000ooooo`00
2ooo00<00?ooool00_oo0`00Q?oo00<00?ooool0P?oo0002ool01000ooooo`002?oo0`001Ooo00<0
0?ooool0QOoo00<00?ooool0Oooo0002ool01000ooooo`002?oo00<00?ooool01Ooo00<00?ooool0
Q_oo00<00?ooool0O_oo0002ool01000ooooo`002?oo00<00?ooool01Ooo00<00?ooool0Qooo00<0
0?ooool0OOoo0003ool20009ool40004ool00`00ooooo`28ool00`00ooooo`1lool001Koo`03003o
oooo08Woo`03003ooooo07_oo`005_oo0P00Rooo00<00?ooool0N_oo000Fool00`00ooooo`2;ool0
0`00ooooo`1iool001Koo`03003ooooo08coo`03003ooooo07Soo`005_oo00<00?ooool0SOoo00<0
0?ooool0Mooo000Fool00`00ooooo`2>ool00`00ooooo`1fool001Koo`03003ooooo08ooo`03003o
oooo07Goo`005_oo00<00?ooool0T?oo00<00?ooool0M?oo000Fool2002Bool00`00ooooo`1cool0
01Koo`03003ooooo09;oo`03003ooooo07;oo`005_oo00<00?ooool0T_oo00<00?ooool0L_oo000F
ool00`00ooooo`2Cool00`00ooooo`1aool001Koo`03003ooooo09Coo`03003ooooo073oo`005_oo
00<00?ooool0UOoo00<00?ooool0Kooo000Fool2002Gool00`00ooooo`1^ool001Koo`03003ooooo
09Koo`03003ooooo06koo`005_oo00<00?ooool0Uooo00<00?ooool0KOoo000Fool00`00ooooo`2H
ool00`00ooooo`1/ool001Koo`03003ooooo09Woo`03003ooooo06_oo`005_oo00<00?ooool0VOoo
00<00?ooool0Jooo000Fool00`00ooooo`2Jool00`00ooooo`1Zool001Koo`8009coo`8000goo`l0
04koo`005_oo00<00?ooool0W?oo00<00?ooool01Ooo1P003ooo1`00Aooo000Fool00`00ooooo`2M
ool7000Lool30014ool001Koo`03003ooooo0<?oo`<0047oo`000ooo0P001?oo00<00?ooool00ooo
0P001Ooo00<00?ooool0a_oo0P00?ooo0002ool01000ooooo`002?oo00@00?ooool000Coo`03003o
oooo0<Soo`<003coo`000_oo00@00?ooool000Soo`04003ooooo0004ool3003;ool2000jool000;o
o`04003ooooo0008ool30005ool00`00ooooo`3=ool2000hool000;oo`04003ooooo0008ool00`00
ooooo`05ool00`00ooooo`3?ool2000fool000;oo`04003ooooo0008ool00`00ooooo`05ool00`00
ooooo`3Aool2000dool000?oo`8000[oo`<000Coo`03003ooooo0=?oo`8003;oo`005_oo00<00?oo
ool0eOoo00<00?ooool0;ooo000Fool2003Gool2000_ool001Koo`03003ooooo0=Soo`03003ooooo
02coo`005_oo00<00?ooool0fOoo00<00?ooool0:ooo000Fool00`00ooooo`3Jool00`00ooooo`0Z
ool001Koo`03003ooooo0=_oo`03003ooooo02Woo`005_oo00<00?ooool0g?oo00<00?ooool0:?oo
000Fool00`00ooooo`3Mool00`00ooooo`0Wool001Koo`800=ooo`03003ooooo02Koo`005_oo00<0
0?ooool0gooo00<00?ooool09Ooo000Fool00`00ooooo`3Pool00`00ooooo`0Tool001Koo`03003o
oooo0>7oo`8002Coo`005_oo00<00?ooool0hooo00<00?ooool08Ooo000Fool00`00ooooo`3Tool0
0`00ooooo`0Pool001Koo`800>Koo`03003ooooo01ooo`005_oo00<00?ooool0iOoo00<00?ooool0
7ooo000Fool00`00ooooo`3Vool00`00ooooo`0Nool001Koo`03003ooooo0>Ooo`03003ooooo01go
o`005_oo00<00?ooool0j?oo00<00?ooool07?oo000Fool00`00ooooo`3Xool00`00ooooo`0Lool0
01Koo`03003ooooo0>Woo`03003ooooo01_oo`005_oo0P00jooo00<00?ooool06_oo000Fool00`00
ooooo`3Zool00`00ooooo`0Jool001Koo`03003ooooo0>_oo`03003ooooo01Woo`005_oo00<00?oo
ool0k?oo00<00?ooool06?oo0003ool20004ool00`00ooooo`03ool00`00ooooo`04ool00`00oooo
o`3]ool00`00ooooo`0Gool000;oo`04003ooooo0009ool00`00ooooo`04ool00`00ooooo`3]ool0
0`00ooooo`0Gool000;oo`04003ooooo000:ool00`00ooooo`03ool3003^ool00`00ooooo`0Fool0
00;oo`04003ooooo000:ool00`00ooooo`03ool00`00ooooo`3_ool00`00ooooo`0Eool000;oo`04
003ooooo000;ool00`00ooooo`02ool00`00ooooo`3_ool00`00ooooo`0Eool000;oo`04003ooooo
0008ool01000ooooo`001?oo00<00?ooool0l?oo00<00?ooool05?oo0003ool20009ool40004ool0
0`00ooooo`3`ool00`00ooooo`0Dool001Koo`03003ooooo0?7oo`03003ooooo01?oo`005_oo0P00
l_oo00<00?ooool04ooo000Fool00`00ooooo`3bool00`00ooooo`0Bool001Koo`03003ooooo0?;o
o`03003ooooo01;oo`005_oo00<00?ooool0looo00<00?ooool04Ooo000Fool00`00ooooo`3dool0
0`00ooooo`0@ool001Koo`03003ooooo0?Coo`03003ooooo013oo`005_oo00<00?ooool0mOoo00<0
0?ooool03ooo000Fool2003fool00`00ooooo`0?ool001Koo`03003ooooo0?Koo`03003ooooo00ko
o`005_oo00<00?ooool0m_oo00<00?ooool03_oo000Fool00`00ooooo`3gool00`00ooooo`0=ool0
01Koo`03003ooooo0?Ooo`03003ooooo00goo`005_oo00<00?ooool0n?oo00<00?ooool03?oo000F
ool2003iool00`00ooooo`0<ool001Koo`03003ooooo0?Woo`03003ooooo00_oo`005_oo00<00?oo
ool0nOoo00<00?ooool02ooo000Fool00`00ooooo`3jool00`00ooooo`0:ool001Koo`03003ooooo
0?[oo`03003ooooo00[oo`005_oo00<00?ooool0nooo00<00?ooool02Ooo000Fool00`00ooooo`3k
ool00`00ooooo`09ool001Koo`800?goo`03003ooooo00Soo`005_oo00<00?ooool0o?oo00<00?oo
ool02?oo000Fool00`00ooooo`3mool00`00ooooo`07ool001Koo`03003ooooo0?goo`03003ooooo
00Ooo`000ooo0P001?oo00<00?ooool00ooo0P001Ooo00<00?ooool0o_oo00<00?ooool01_oo0002
ool01000ooooo`002?oo00@00?ooool000Coo`03003ooooo0?koo`03003ooooo00Koo`000_oo00@0
0?ooool000Soo`04003ooooo0004ool3003oool8ool000;oo`04003ooooo0009ool20005ool00`00
ooooo`3oool8ool000;oo`04003ooooo0008ool01000ooooo`001?oo00<00?ooool0oooo2?oo0002
ool01000ooooo`002?oo00@00?ooool00?ooo`ooo`000ooo0P002_oo0P00oooo4?oo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0884228, 0.261369, \
0.00389745, 0.00312361}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[23]="]
}, Open  ]],

Cell["\<\
Despite the fact that a solution has been obtained, it is not clear \
whether it has been obtained efficiently.\
\>", "Text"],

Cell["\<\
The following example shows that the crossing of the discontinuity \
presents difficulties for the numerical solver.\
\>", "Text"],

Cell["\<\
This defines a function that displays the mesh points of the \
integration together with the number of integration steps that are \
taken.\
\>", "MathCaption"],

Cell[BoxData[
    \(\(StepPlot[data_, 
          opts___?OptionQ] := \[IndentingNewLine]Module[{sdata}, \
\[IndentingNewLine]sdata\  = \ 
            Transpose[{data, 
                Range[Length[data]]}]; \[IndentingNewLine]ListPlot[sdata, 
            opts, Axes \[Rule] False, Frame \[Rule] True, 
            PlotRange \[Rule] All]\[IndentingNewLine]];\)\)], "Input",
  CellLabel->"In[24]:="],

Cell[TextData[{
  "As the integration passes the discontinuity (near ",
  Cell[BoxData[
      \(TraditionalForm\`0.6\)]],
  " in value) the integration method runs into difficulty and a large number \
of small steps are taken - a number of rejected steps can also sometimes be \
observed."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(StepPlot[data]\)], "Input",
  CellLabel->"In[25]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0237127 0.952478 0.00916223 0.00555287 [
[.02371 -0.0125 -3 -9 ]
[.02371 -0.0125 3 0 ]
[.21421 -0.0125 -9 -9 ]
[.21421 -0.0125 9 0 ]
[.4047 -0.0125 -9 -9 ]
[.4047 -0.0125 9 0 ]
[.5952 -0.0125 -9 -9 ]
[.5952 -0.0125 9 0 ]
[.78569 -0.0125 -9 -9 ]
[.78569 -0.0125 9 0 ]
[.97619 -0.0125 -3 -9 ]
[.97619 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .00916 -6 -4.5 ]
[-0.0125 .00916 0 4.5 ]
[-0.0125 .12022 -12 -4.5 ]
[-0.0125 .12022 0 4.5 ]
[-0.0125 .23128 -12 -4.5 ]
[-0.0125 .23128 0 4.5 ]
[-0.0125 .34233 -12 -4.5 ]
[-0.0125 .34233 0 4.5 ]
[-0.0125 .45339 -12 -4.5 ]
[-0.0125 .45339 0 4.5 ]
[-0.0125 .56445 -18 -4.5 ]
[-0.0125 .56445 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02371 0 m
.02371 .00625 L
s
[(0)] .02371 -0.0125 0 1 Mshowa
.21421 0 m
.21421 .00625 L
s
[(0.2)] .21421 -0.0125 0 1 Mshowa
.4047 0 m
.4047 .00625 L
s
[(0.4)] .4047 -0.0125 0 1 Mshowa
.5952 0 m
.5952 .00625 L
s
[(0.6)] .5952 -0.0125 0 1 Mshowa
.78569 0 m
.78569 .00625 L
s
[(0.8)] .78569 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(1)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07134 0 m
.07134 .00375 L
s
.11896 0 m
.11896 .00375 L
s
.16658 0 m
.16658 .00375 L
s
.26183 0 m
.26183 .00375 L
s
.30946 0 m
.30946 .00375 L
s
.35708 0 m
.35708 .00375 L
s
.45233 0 m
.45233 .00375 L
s
.49995 0 m
.49995 .00375 L
s
.54758 0 m
.54758 .00375 L
s
.64282 0 m
.64282 .00375 L
s
.69045 0 m
.69045 .00375 L
s
.73807 0 m
.73807 .00375 L
s
.83332 0 m
.83332 .00375 L
s
.88094 0 m
.88094 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .00916 m
.00625 .00916 L
s
[(0)] -0.0125 .00916 1 0 Mshowa
0 .12022 m
.00625 .12022 L
s
[(20)] -0.0125 .12022 1 0 Mshowa
0 .23128 m
.00625 .23128 L
s
[(40)] -0.0125 .23128 1 0 Mshowa
0 .34233 m
.00625 .34233 L
s
[(60)] -0.0125 .34233 1 0 Mshowa
0 .45339 m
.00625 .45339 L
s
[(80)] -0.0125 .45339 1 0 Mshowa
0 .56445 m
.00625 .56445 L
s
[(100)] -0.0125 .56445 1 0 Mshowa
.125 Mabswid
0 .03693 m
.00375 .03693 L
s
0 .06469 m
.00375 .06469 L
s
0 .09246 m
.00375 .09246 L
s
0 .14798 m
.00375 .14798 L
s
0 .17575 m
.00375 .17575 L
s
0 .20351 m
.00375 .20351 L
s
0 .25904 m
.00375 .25904 L
s
0 .28681 m
.00375 .28681 L
s
0 .31457 m
.00375 .31457 L
s
0 .3701 m
.00375 .3701 L
s
0 .39786 m
.00375 .39786 L
s
0 .42563 m
.00375 .42563 L
s
0 .48116 m
.00375 .48116 L
s
0 .50892 m
.00375 .50892 L
s
0 .53668 m
.00375 .53668 L
s
0 .59221 m
.00375 .59221 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.02371 .61178 m
.02371 .61803 L
s
.21421 .61178 m
.21421 .61803 L
s
.4047 .61178 m
.4047 .61803 L
s
.5952 .61178 m
.5952 .61803 L
s
.78569 .61178 m
.78569 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.07134 .61428 m
.07134 .61803 L
s
.11896 .61428 m
.11896 .61803 L
s
.16658 .61428 m
.16658 .61803 L
s
.26183 .61428 m
.26183 .61803 L
s
.30946 .61428 m
.30946 .61803 L
s
.35708 .61428 m
.35708 .61803 L
s
.45233 .61428 m
.45233 .61803 L
s
.49995 .61428 m
.49995 .61803 L
s
.54758 .61428 m
.54758 .61803 L
s
.64282 .61428 m
.64282 .61803 L
s
.69045 .61428 m
.69045 .61803 L
s
.73807 .61428 m
.73807 .61803 L
s
.83332 .61428 m
.83332 .61803 L
s
.88094 .61428 m
.88094 .61803 L
s
.92857 .61428 m
.92857 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .00916 m
1 .00916 L
s
.99375 .12022 m
1 .12022 L
s
.99375 .23128 m
1 .23128 L
s
.99375 .34233 m
1 .34233 L
s
.99375 .45339 m
1 .45339 L
s
.99375 .56445 m
1 .56445 L
s
.125 Mabswid
.99625 .03693 m
1 .03693 L
s
.99625 .06469 m
1 .06469 L
s
.99625 .09246 m
1 .09246 L
s
.99625 .14798 m
1 .14798 L
s
.99625 .17575 m
1 .17575 L
s
.99625 .20351 m
1 .20351 L
s
.99625 .25904 m
1 .25904 L
s
.99625 .28681 m
1 .28681 L
s
.99625 .31457 m
1 .31457 L
s
.99625 .3701 m
1 .3701 L
s
.99625 .39786 m
1 .39786 L
s
.99625 .42563 m
1 .42563 L
s
.99625 .48116 m
1 .48116 L
s
.99625 .50892 m
1 .50892 L
s
.99625 .53668 m
1 .53668 L
s
.99625 .59221 m
1 .59221 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.008 w
.6175 .40342 Mdot
.6175 .39786 Mdot
.61751 .41452 Mdot
.6175 .40897 Mdot
.6175 .38676 Mdot
.6175 .3812 Mdot
.6175 .39231 Mdot
.61755 .44229 Mdot
.61753 .43673 Mdot
.61759 .45339 Mdot
.61757 .44784 Mdot
.61751 .42563 Mdot
.61751 .42007 Mdot
.61751 .43118 Mdot
.6175 .33123 Mdot
.61749 .32568 Mdot
.6175 .34233 Mdot
.6175 .33678 Mdot
.61746 .31457 Mdot
.61746 .30902 Mdot
.61749 .32012 Mdot
.6175 .3701 Mdot
.6175 .36455 Mdot
.6175 .37565 Mdot
.6175 .35344 Mdot
.6175 .34789 Mdot
.6175 .35899 Mdot
.8054 .55334 Mdot
.78347 .54779 Mdot
.84926 .56445 Mdot
.82733 .5589 Mdot
.7396 .53668 Mdot
.72097 .53113 Mdot
.76154 .54224 Mdot
.95069 .59221 Mdot
.93287 .58666 Mdot
.97619 .60332 Mdot
.96344 .59777 Mdot
.89312 .57555 Mdot
.87119 .57 Mdot
.91506 .58111 Mdot
.62023 .48116 Mdot
.61834 .4756 Mdot
.62401 .49226 Mdot
.62212 .48671 Mdot
.61796 .4645 Mdot
.61778 .45894 Mdot
.61815 .47005 Mdot
.6837 .52003 Mdot
.66506 .51447 Mdot
.70233 .52558 Mdot
.62779 .50337 Mdot
.6259 .49781 Mdot
.64643 .50892 Mdot
.27445 .10911 Mdot
.24423 .10356 Mdot
.33487 .12022 Mdot
.30466 .11467 Mdot
.1838 .09246 Mdot
.15359 .0869 Mdot
.21402 .09801 Mdot
.48073 .14798 Mdot
.45573 .14243 Mdot
.53073 .15909 Mdot
.50573 .15354 Mdot
.3953 .13133 Mdot
.36509 .12577 Mdot
.42552 .13688 Mdot
.03227 .03693 Mdot
.02948 .03137 Mdot
.0496 .04803 Mdot
.04093 .04248 Mdot
.02391 .02027 Mdot
.02381 .01472 Mdot
.02669 .02582 Mdot
.11893 .0758 Mdot
.10159 .07024 Mdot
.13626 .08135 Mdot
.06693 .05914 Mdot
.05827 .05359 Mdot
.08426 .06469 Mdot
.61745 .25904 Mdot
.61745 .25349 Mdot
.61745 .27015 Mdot
.61745 .26459 Mdot
.61737 .24238 Mdot
.61716 .23683 Mdot
.61741 .24794 Mdot
.61745 .29791 Mdot
.61745 .29236 Mdot
.61746 .30346 Mdot
.61745 .28125 Mdot
.61745 .2757 Mdot
.61745 .28681 Mdot
.61186 .18685 Mdot
.60667 .1813 Mdot
.61706 .19796 Mdot
.61705 .19241 Mdot
.58073 .1702 Mdot
.55573 .16464 Mdot
.60148 .17575 Mdot
.61712 .22572 Mdot
.6171 .22017 Mdot
.61714 .23128 Mdot
.61708 .20907 Mdot
.61707 .20351 Mdot
.61709 .21462 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[25]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool001[oo`8002_oo`8000Coo`03003ooooo00;oo`@002?oo`8000Coo`03003ooooo00Coo`03003o
oooo02?oo`8000Coo`03003ooooo00?oo`8002Coo`8000Coo`03003ooooo00?oo`8002[oo`@000Ko
o`006Ooo00@00?ooool002Woo`04003ooooo0008ool00`00ooooo`0Sool01000ooooo`002_oo00<0
0?ooool08_oo00@00?ooool000Soo`04003ooooo000Rool01000ooooo`002?oo00@00?ooool002_o
o`03003ooooo00Goo`006Ooo00@00?ooool002Woo`04003ooooo0009ool00`00ooooo`0Rool01000
ooooo`001ooo1@008ooo00@00?ooool000Soo`04003ooooo000Rool01000ooooo`002?oo00@00?oo
ool002_oo`03003ooooo00Goo`006Ooo00@00?ooool002Woo`04003ooooo000:ool00`00ooooo`0Q
ool01000ooooo`001ooo00@00?ooool002Coo`04003ooooo0008ool3000Sool01000ooooo`002Ooo
0P00;?oo00<00?ooool01Ooo000Iool01000ooooo`00:Ooo00@00?ooool000_oo`03003ooooo023o
o`04003ooooo0008ool00`00ool0000Tool01000ooooo`002?oo00<00?ooool08ooo00@00?ooool0
00Soo`04003ooooo000[ool00`00ooooo`05ool001Woo`04003ooooo000Yool01000ooooo`002?oo
00@00?ooool002;oo`04003ooooo0009ool2000Tool01000ooooo`002?oo00<00?ooool08ooo00@0
0?ooool000Soo`04003ooooo000Yool30007ool001[oo`8002_oo`8000[oo`8002Coo`8000_oo`03
003ooooo02?oo`8000[oo`<002?oo`8000[oo`8002coo`03003ooooo00Goo`00oooo8Ooo003ooolQ
ool00?ooob7oo`005Oooo`002`000Ooo000>ool20005ool00`00ooooo`03ool00`00ooooo`0:ool0
0`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`0:ool0
0`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`0:ool00`00ooooo`09ool0
0`00ooooo`0:ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`09ool0
0`00ooooo`0:ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`03ool1
0001ool000goo`04003ooooo0004ool00`00ooooo`03ool00`00ooooo`0`ool00`00ooooo`0_ool0
0`00ooooo`0`ool00`00ooooo`0_ool00`00ooooo`0`ool00`00ooooo`03ool10001ool000goo`04
003ooooo0004ool2003oool7ool20001ool000goo`04003ooooo0004ool00`00ooooo`03ool2003o
ool2ool10001ool000goo`04003ooooo0004ool00`00ooooo`03ool2003oool2ool10001ool000go
o`04003ooooo0004ool00`00ooooo`04ool00`00ooooo`3oool10001ool000koo`8000Goo`03003o
oooo00?oo`800?ooo`;oo`40007oo`005Ooo00<00?ooool01?oo0P00oooo0Ooo0@000Ooo000Eool0
0`00ooooo`05ool2003oool10001ool001Goo`8000Ooo`03003ooooo0?coo`80007oo`005Ooo00<0
0?ooool01ooo0P00oOoo0@000Ooo000Eool00`00ooooo`08ool3003kool10001ool001Goo`03003o
oooo00[oo`03003ooooo0?Woo`40007oo`005Ooo00<00?ooool03?oo0P00n?oo0@000Ooo000Eool0
0`00ooooo`0=ool3003fool10001ool001Goo`03003ooooo00ooo`03003ooooo0?Coo`40007oo`00
5Ooo0P005?oo0P00l?oo0P000Ooo000Eool00`00ooooo`0Dool00`00ooooo`02003]ool10001ool0
01Goo`03003ooooo01Soo`03003ooooo0>_oo`40007oo`005Ooo00<00?ooool07?oo0P00j?oo0@00
0Ooo000Eool00`00ooooo`0Mool00`00ooooo`02003Tool10001ool001Goo`03003ooooo027oo`03
003ooooo0>;oo`40007oo`005Ooo00<00?ooool09Ooo0P00gooo0@000Ooo000Eool00`00ooooo`0V
ool00`00ooooo`04ool2003Gool10001ool001Goo`8002ooo`03003ooooo0=Coo`80007oo`005Ooo
00<00?ooool0=Ooo0P00cooo0@000Ooo000Eool00`00ooooo`0fool00`00ooooo`04ool20037ool1
0001ool001Goo`03003ooooo03koo`03003ooooo0<Goo`40007oo`005Ooo00<00?ooool0AOoo0P00
_ooo0@000Ooo0007ool40003ool20005ool00`00ooooo`16ool00`00ooooo`04ool2002gool10001
ool000Ooo`03003ooooo00?oo`04003ooooo0004ool00`00ooooo`1>ool00`00ooooo`2eool10001
ool000Soo`03003ooooo00;oo`04003ooooo0004ool2001Fool2002^ool20001ool000Woo`05003o
ooooool00002ool00`00ooooo`02ool00`00ooooo`1Fool00`00ooooo`04ool2002Wool10001ool0
00[oo`04003ooooo0002ool00`00ooooo`02ool00`00ooooo`1Nool00`00ooooo`2Uool10001ool0
00Ooo`04003ooooo0002ool01000ooooo`001?oo00<00?ooool0IOoo0P00Wooo0@000Ooo0008ool2
0004ool20005ool00`00ooooo`1Vool00`00ooooo`04ool2002Gool10001ool001Goo`03003ooooo
06koo`03003ooooo09Goo`40007oo`005Ooo00<00?ooool0MOoo0P00Sooo0@000Ooo000Eool2001g
ool00`00ooooo`03ool20027ool20001ool001Goo`03003ooooo07goo`03003ooooo00?oo`80087o
o`40007oo`005Ooo00<00?ooool0Q?oo00<00?ooool0Oooo0@000Ooo000Eool00`00ooooo`29ool2
001kool10001ool001Goo`03003ooooo08[oo`03003ooooo00?oo`8007Coo`40007oo`005Ooo00<0
0?ooool0TOoo00<00?ooool0L_oo0@000Ooo000Eool00`00ooooo`2Fool2001^ool10001ool001Go
o`03003ooooo09Ooo`03003ooooo00;oo`8006Soo`40007oo`005Ooo0P00W_oo00<00?ooool0IOoo
0P000Ooo000Eool00`00ooooo`2Mool2001Wool10001ool001Goo`03003ooooo09koo`<006Goo`40
007oo`005Ooo00<00?ooool0X?oo00<00?ooool0Hooo0@000Ooo000Eool00`00ooooo`2Pool2001T
ool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0XOoo00<00?oo
ool0H_oo0@000Ooo000Eool2002Qool2001Sool20001ool001Goo`03003ooooo0:3oo`8006Coo`40
007oo`005Ooo00<00?ooool0XOoo00<00?ooool0H_oo0@000Ooo000Eool00`00ooooo`2Pool2001T
ool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0XOoo00<00?oo
ool0H_oo0@000Ooo0009ool00`00ooooo`02ool20005ool00`00ooooo`2Pool2001Tool10001ool0
00Woo`05003oooooool00002ool00`00ooooo`02ool00`00ooooo`2Pool2001Tool10001ool000Ko
o`D000;oo`04003ooooo0004ool2002Rool00`00ooooo`1Qool20001ool000Koo`04003ooooo0003
ool01000ooooo`001?oo00<00?ooool0X?oo0P00I?oo0@000Ooo0007ool00`00ool00003ool01000
ooooo`001?oo00<00?ooool0X?oo0P00I?oo0@000Ooo0008ool20003ool01000ooooo`001?oo00<0
0?ooool0XOoo00<00?ooool0H_oo0@000Ooo0009ool00`00ooooo`02ool20005ool00`00ooooo`2P
ool2001Tool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool2002Rool00`00ooooo`1Qool20001ool001Goo`03003ooooo0:3oo`80
06Coo`40007oo`005Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00
ooooo`1Rool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00ooooo`1Rool10001ool001Goo`800:7oo`80
06?oo`80007oo`005Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00
ooooo`1Rool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00ooooo`1Rool10001ool001Goo`03003ooooo
0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo000Eool2002Rool00`00
ooooo`1Qool20001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00ooooo`1Rool10001ool001Goo`03003ooooo
0:3oo`8006Coo`40007oo`002?oo0P001?oo0P001Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo0007
ool01000ooooo`000_oo00@00?ooool000Coo`03003ooooo0:7oo`03003ooooo06;oo`40007oo`00
1ooo00@00?ooool000;oo`04003ooooo0004ool2002Qool2001Sool20001ool000Ooo`<000?oo`04
003ooooo0004ool00`00ooooo`2Pool2001Tool10001ool000Ooo`03003ooooo00?oo`04003ooooo
0004ool00`00ooooo`2Qool00`00ooooo`1Rool10001ool000Ooo`03003ooooo00?oo`04003ooooo
0004ool00`00ooooo`2Pool2001Tool10001ool000Soo`<000?oo`8000Goo`03003ooooo0:3oo`80
06Coo`40007oo`005Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00
ooooo`1Rool10001ool001Goo`800:7oo`8006?oo`80007oo`005Ooo00<00?ooool0X?oo0P00I?oo
0@000Ooo000Eool00`00ooooo`2Qool00`00ooooo`1Rool10001ool001Goo`03003ooooo0:3oo`80
06Coo`40007oo`005Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00
ooooo`1Rool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool2002Rool00`00ooooo`1Qool20001ool001Goo`03003ooooo0:3oo`80
06Coo`40007oo`005Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00
ooooo`1Rool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00ooooo`1Rool10001ool001Goo`800:7oo`80
06?oo`80007oo`005Ooo00<00?ooool0X?oo0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00
ooooo`1Rool10001ool001Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool00`00ooooo`1Rool10001ool000Soo`8000Coo`80
00Goo`03003ooooo0:3oo`8006Coo`40007oo`001ooo00@00?ooool000;oo`04003ooooo0004ool0
0`00ooooo`2Pool2001Tool10001ool000Ooo`04003ooooo0002ool01000ooooo`001?oo0P00X_oo
00<00?ooool0HOoo0P000Ooo0008ool20003ool01000ooooo`001?oo00<00?ooool0X?oo0P00I?oo
0@000Ooo0007ool01000ooooo`000_oo00@00?ooool000Coo`03003ooooo0:3oo`8006Coo`40007o
o`001ooo00@00?ooool000;oo`04003ooooo0004ool00`00ooooo`2Qool00`00ooooo`1Rool10001
ool000Soo`8000Coo`8000Goo`03003ooooo0:3oo`8006Coo`40007oo`005Ooo00<00?ooool0X?oo
0P00I?oo0@000Ooo000Eool00`00ooooo`2Qool2001Sool10001ool001Goo`800:?oo`03003ooooo
063oo`80007oo`005Ooo00<00?ooool0XOoo0P00Hooo0@000Ooo000Eool00`00ooooo`2Rool2001R
ool10001ool001Goo`03003ooooo0:?oo`03003ooooo063oo`40007oo`005Ooo00<00?ooool0X_oo
0P00H_oo0@000Ooo000Eool00`00ooooo`2Sool2001Qool10001ool001Goo`03003ooooo0:Coo`03
003ooooo05ooo`40007oo`005Ooo0P00ZOoo0P00Fooo0P000Ooo000Eool00`00ooooo`2Yool01000
oooooooo0P00Eooo0@000Ooo000Eool00`00ooooo`2^ool00`00ooooo`1Eool10001ool001Goo`03
003ooooo0;;oo`8005;oo`40007oo`005Ooo00<00?ooool0/ooo00@00?ooooooo`8004goo`40007o
o`005Ooo00<00?ooool0^?oo00<00?ooool0Booo0@000Ooo000Eool00`00ooooo`2lool20018ool1
0001ool001Goo`03003ooooo0;goo`04003oooooool20013ool10001ool001Goo`800<?oo`03003o
oooo043oo`80007oo`005Ooo00<00?ooool0a_oo0P00?_oo0@000Ooo000Eool00`00ooooo`37ool0
0`00ooooo`02ool2000hool10001ool001Goo`03003ooooo0<goo`03003ooooo03Koo`40007oo`00
5Ooo00<00?ooool0d_oo0P00<_oo0@000Ooo00000ooo000000020003ool20004ool20005ool00`00
ooooo`3Cool00`00ooooo`02ool2000/ool10001ool000?oo`05003oooooool00002ool01000oooo
o`000_oo00<00?ooool00_oo00<00?ooool0fOoo00<00?ooool0:_oo0@000Ooo0003ool01@00oooo
oooo00000_oo00@00?ooool000;oo`03003ooooo00;oo`800=ooo`8002Goo`80007oo`000ooo00D0
0?ooooooo`0000;oo`04003ooooo0002ool00`00ooooo`02ool00`00ooooo`3Oool00`00ooooo`02
ool2000Pool10001ool000?oo`05003oooooool00002ool01000ooooo`000_oo00<00?ooool00_oo
00<00?ooool0iOoo00<00?ooool07_oo0@000Ooo00001?oo000000000ooo00@00?ooool000;oo`04
003ooooo0004ool00`00ooooo`3Yool2000Kool10001ool000?oo`03003ooooo00;oo`8000Coo`80
00Goo`03003ooooo0>[oo`03003ooooo00;oo`8001Goo`40007oo`005Ooo00<00?ooool0l?oo00@0
0?ooooooo`80013oo`40007oo`005Ooo00<00?ooool0mOoo00<00?ooool03_oo0@000Ooo000Eool2
003jool2000:ool20001ool001Goo`03003ooooo0?[oo`04003oo`000008ool10001ool001Goo`03
003ooooo0?goo`03003ooooo00Koo`40007oo`005Ooo00<00?ooool0oooo0P001Ooo0@000Ooo000E
ool00`00ooooo`3oool1ool00`00ooooo`03ool10001ool001Goo`03003ooooo0?ooo`Ooo`40007o
o`005Ooo00<00?ooool00ooo00<00?ooool02_oo00<00?ooool02Ooo00<00?ooool02_oo00<00?oo
ool02_oo00<00?ooool02Ooo00<00?ooool02_oo00<00?ooool02_oo00<00?ooool02Ooo00<00?oo
ool02_oo00<00?ooool02_oo00<00?ooool02Ooo00<00?ooool02_oo00<00?ooool02_oo00<00?oo
ool02Ooo00<00?ooool02_oo00<00?ooool02Ooo00<00?ooool02_oo00<00?ooool02_oo00<00?oo
ool02Ooo00<00?ooool02_oo00<00?ooool00ooo0@000Ooo000Eoooo000;0001ool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.10919, -10.1735, \
0.00395361, 0.678159}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[25]="]
}, Open  ]],

Cell["\<\
One of the most efficient methods of crossing a discontinuity is to \
break the integration by restarting at the point of discontinuity.\
\>", \
"Text"],

Cell[TextData[{
  "The following example shows how to use the ",
  StyleBox["EventLocator", "MR"],
  " method to accomplish this."
}], "Text"],

Cell["\<\
This numerically integrates the first part of the system up to the \
point of discontinuity. The switching function is given as the event. The \
direction of the event is restricted to a change from negative to positive. \
When the event is found, the solution and the time of the event are stored by \
the event action.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(\(system1\  = \ {\(y'\)[t] \[Equal] t^2 + 2  y[t]^2, 
          y[t0] \[Equal] ics0};\)\(\[IndentingNewLine]\)
    \)\), "\[IndentingNewLine]", 
    \(\(data1\  = \ \(Reap[
            sol1\ \  = \ 
              y[t]\  /. \ First[
                  NDSolve[system1, y, {t, t0, 1}, 
                    Method \[Rule] {"\<EventLocator\>", "\<Event\>" -> event, 
                        Direction \[Rule] 1, 
                        EventAction \[RuleDelayed] 
                          Throw[t1\  = \ t; \ 
                            ics1\  = \ y[t];\ , "\<StopIntegration\>"], 
                        Method \[Rule] odemethod}, MaxStepFraction \[Rule] 1, 
                    StepMonitor \[RuleDelayed] 
                      Sow[t]]]\[IndentingNewLine]]\)[\([2, 
            1]\)];\)\), "\[IndentingNewLine]", 
    \(sol1\)}], "Input",
  CellLabel->"In[26]:="],

Cell[BoxData[
    RowBox[{
      TagBox[\(InterpolatingFunction[{{0.`, 0.6234178950846722`}}, "<>"]\),
        False,
        Editable->False], "[", "t", "]"}]], "Output",
  CellLabel->"Out[28]="]
}, Open  ]],

Cell[TextData[{
  "Using the discontinuity found by the ",
  StyleBox["EventLocator", "MR"],
  " method as a new initial condition, the integration can now be continued."
}], "Text"],

Cell["\<\
This defines a system and initial condition, solves the system \
numerically and collects the data used for the mesh points.\
\>", \
"MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(\(system2\  = \ {\(y'\)[t] \[Equal] 2  t^2 + 3  y[t]^2 - 2, 
          y[t1] \[Equal] ics1};\)\(\[IndentingNewLine]\)
    \)\), "\[IndentingNewLine]", 
    \(\(data2\  = \ \(Reap[\[IndentingNewLine]sol2\  = \ 
              y[t]\  /. \ First[
                  NDSolve[system2, y, {t, t1, 1}, Method \[Rule] odemethod, 
                    MaxStepFraction \[Rule] 1, 
                    StepMonitor \[RuleDelayed] 
                      Sow[t]]]\[IndentingNewLine]]\)[\([2, 
            1]\)];\)\), "\[IndentingNewLine]", 
    \(sol2\)}], "Input",
  CellLabel->"In[29]:="],

Cell[BoxData[
    RowBox[{
      TagBox[\(InterpolatingFunction[{{0.6234178950846722`, 1.`}}, "<>"]\),
        False,
        Editable->False], "[", "t", "]"}]], "Output",
  CellLabel->"Out[31]="]
}, Open  ]],

Cell["\<\
A plot of the two solutions is very similar to that obtained by \
solving the entire system in one go.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(evsol\  = \ 
      Plot[If[t\  \[LessEqual] \ t1, sol1, sol2], {t, 0, 1}]\)], "Input",
  CellLabel->"In[32]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 -0.34178 1.18832 [
[.21429 .00222 -9 -9 ]
[.21429 .00222 9 0 ]
[.40476 .00222 -9 -9 ]
[.40476 .00222 9 0 ]
[.59524 .00222 -9 -9 ]
[.59524 .00222 9 0 ]
[.78571 .00222 -9 -9 ]
[.78571 .00222 9 0 ]
[.97619 .00222 -3 -9 ]
[.97619 .00222 3 0 ]
[.01131 .13355 -18 -4.5 ]
[.01131 .13355 0 4.5 ]
[.01131 .25238 -18 -4.5 ]
[.01131 .25238 0 4.5 ]
[.01131 .37121 -18 -4.5 ]
[.01131 .37121 0 4.5 ]
[.01131 .49004 -18 -4.5 ]
[.01131 .49004 0 4.5 ]
[.01131 .60887 -18 -4.5 ]
[.01131 .60887 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .01472 m
.21429 .02097 L
s
[(0.2)] .21429 .00222 0 1 Mshowa
.40476 .01472 m
.40476 .02097 L
s
[(0.4)] .40476 .00222 0 1 Mshowa
.59524 .01472 m
.59524 .02097 L
s
[(0.6)] .59524 .00222 0 1 Mshowa
.78571 .01472 m
.78571 .02097 L
s
[(0.8)] .78571 .00222 0 1 Mshowa
.97619 .01472 m
.97619 .02097 L
s
[(1)] .97619 .00222 0 1 Mshowa
.125 Mabswid
.07143 .01472 m
.07143 .01847 L
s
.11905 .01472 m
.11905 .01847 L
s
.16667 .01472 m
.16667 .01847 L
s
.2619 .01472 m
.2619 .01847 L
s
.30952 .01472 m
.30952 .01847 L
s
.35714 .01472 m
.35714 .01847 L
s
.45238 .01472 m
.45238 .01847 L
s
.5 .01472 m
.5 .01847 L
s
.54762 .01472 m
.54762 .01847 L
s
.64286 .01472 m
.64286 .01847 L
s
.69048 .01472 m
.69048 .01847 L
s
.7381 .01472 m
.7381 .01847 L
s
.83333 .01472 m
.83333 .01847 L
s
.88095 .01472 m
.88095 .01847 L
s
.92857 .01472 m
.92857 .01847 L
s
.25 Mabswid
0 .01472 m
1 .01472 L
s
.02381 .13355 m
.03006 .13355 L
s
[(0.4)] .01131 .13355 1 0 Mshowa
.02381 .25238 m
.03006 .25238 L
s
[(0.5)] .01131 .25238 1 0 Mshowa
.02381 .37121 m
.03006 .37121 L
s
[(0.6)] .01131 .37121 1 0 Mshowa
.02381 .49004 m
.03006 .49004 L
s
[(0.7)] .01131 .49004 1 0 Mshowa
.02381 .60887 m
.03006 .60887 L
s
[(0.8)] .01131 .60887 1 0 Mshowa
.125 Mabswid
.02381 .03848 m
.02756 .03848 L
s
.02381 .06225 m
.02756 .06225 L
s
.02381 .08601 m
.02756 .08601 L
s
.02381 .10978 m
.02756 .10978 L
s
.02381 .15731 m
.02756 .15731 L
s
.02381 .18108 m
.02756 .18108 L
s
.02381 .20485 m
.02756 .20485 L
s
.02381 .22861 m
.02756 .22861 L
s
.02381 .27615 m
.02756 .27615 L
s
.02381 .29991 m
.02756 .29991 L
s
.02381 .32368 m
.02756 .32368 L
s
.02381 .34744 m
.02756 .34744 L
s
.02381 .39498 m
.02756 .39498 L
s
.02381 .41874 m
.02756 .41874 L
s
.02381 .44251 m
.02756 .44251 L
s
.02381 .46628 m
.02756 .46628 L
s
.02381 .51381 m
.02756 .51381 L
s
.02381 .53757 m
.02756 .53757 L
s
.02381 .56134 m
.02756 .56134 L
s
.02381 .58511 m
.02756 .58511 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .01472 m
.02498 .01498 L
.02615 .01524 L
.02848 .01577 L
.03316 .01683 L
.0425 .01897 L
.06277 .02371 L
.08169 .0283 L
.10025 .03296 L
.12037 .03823 L
.13915 .04338 L
.15951 .04924 L
.17949 .05531 L
.19813 .06128 L
.21834 .06813 L
.23721 .0749 L
.25571 .08193 L
.27578 .09004 L
.29451 .09809 L
.31481 .10738 L
.33473 .11712 L
.35332 .1268 L
.37348 .13801 L
.39229 .14918 L
.41268 .16212 L
.4327 .17572 L
.45137 .18928 L
.47162 .20501 L
.49052 .22072 L
.50905 .23716 L
.52915 .25625 L
.54791 .27532 L
.56825 .29748 L
.58821 .32087 L
.60683 .34425 L
.61218 .35128 L
.61486 .35485 L
.6162 .35665 L
.61754 .35845 L
s
.61754 .35845 m
.61871 .35819 L
.61988 .35793 L
.62221 .35742 L
.62689 .35645 L
.63623 .35467 L
.64637 .35301 L
.65143 .35229 L
.6565 .35164 L
.66596 .35065 L
.67069 .35026 L
.67542 .34995 L
.67774 .34982 L
.68006 .34971 L
.68238 .34962 L
.6847 .34955 L
.68586 .34952 L
.68702 .34949 L
.68818 .34948 L
.68934 .34946 L
.6905 .34945 L
.69166 .34945 L
.69282 .34945 L
.69398 .34945 L
.69523 .34946 L
.69649 .34948 L
.69775 .3495 L
.69901 .34953 L
.70027 .34957 L
.70152 .34961 L
.70404 .34971 L
.70655 .34984 L
.70907 .34999 L
.7141 .35037 L
.71645 .35058 L
.7188 .35081 L
.72349 .35135 L
.72819 .35199 L
.73288 .35272 L
.74306 .35465 L
.75324 .35707 L
.76323 .35995 L
.77322 .36336 L
.79186 .3712 L
.81207 .38207 L
.83094 .39467 L
.84944 .40955 L
.86951 .42881 L
.88824 .45003 L
.90854 .47696 L
Mistroke
.92847 .50785 L
.94705 .54112 L
.96721 .58273 L
.9717 .59286 L
.97395 .59805 L
.97507 .60067 L
.97619 .60332 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[32]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool004?oo`8000Coo`03003ooooo00;oo`@002Coo`8000Coo`03003ooooo00Coo`03003ooooo02?o
o`8000Coo`03003ooooo00?oo`8002Goo`8000Coo`03003ooooo00?oo`8002_oo`@000Ooo`00@_oo
00@00?ooool000Soo`03003ooooo02Coo`04003ooooo000:ool00`00ooooo`0Rool01000ooooo`00
2?oo00@00?ooool002?oo`04003ooooo0008ool01000ooooo`00;?oo00<00?ooool01_oo0012ool0
1000ooooo`002Ooo00<00?ooool08ooo00@00?ooool000Ooo`D002?oo`04003ooooo0008ool01000
ooooo`008ooo00@00?ooool000Soo`04003ooooo000/ool00`00ooooo`06ool004;oo`04003ooooo
000:ool00`00ooooo`0Rool01000ooooo`001ooo00@00?ooool002Coo`04003ooooo0008ool3000T
ool01000ooooo`002Ooo0P00;Ooo00<00?ooool01_oo0012ool01000ooooo`002ooo00<00?ooool0
8Ooo00@00?ooool000Soo`03003oo`0002Coo`04003ooooo0008ool00`00ooooo`0Tool01000oooo
o`002?oo00@00?ooool002coo`03003ooooo00Koo`00@_oo00@00?ooool000Soo`04003ooooo000S
ool01000ooooo`002Ooo0P009?oo00@00?ooool000Soo`03003ooooo02Coo`04003ooooo0008ool0
1000ooooo`00:_oo0`002?oo000Fool00`00ooooo`0Zool2000:ool2000Uool2000;ool00`00oooo
o`0Sool2000:ool3000Tool2000:ool2000]ool00`00ooooo`06ool001Koo`03003ooooo0?ooo`So
o`005_oo00<00?ooool0oooo2?oo000Fool00`00ooooo`3oool8ool0013oool000l000;oo`005_oo
00@00?ooooooo`@000Goo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00Woo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00Woo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00[oo`03003ooooo00Woo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo00[oo`03003ooooo
00[oo`03003ooooo00Koo`005_oo00<00?ooool01Ooo1@009ooo00<00?ooool0<?oo00<00?ooool0
<?oo00<00?ooool0<?oo00<00?ooool0<Ooo00<00?ooool01_oo000Fool00`00ooooo`0:ool2003k
ool001Koo`03003ooooo00coo`@00?Ooo`005_oo00<00?ooool04?oo1@00l_oo000Fool2000Fool4
003^ool001Koo`03003ooooo01Woo`800>coo`005_oo00<00?ooool06ooo1000j?oo000Fool00`00
ooooo`0Oool5003Sool001Koo`03003ooooo02Coo`800>7oo`005_oo00<00?ooool09_oo1000gOoo
000Fool00`00ooooo`0Zool4003Iool001Koo`8002ooo`<00=Koo`005_oo00<00?ooool0<Ooo0`00
dooo000Fool00`00ooooo`0dool2003Aool001Koo`03003ooooo03Koo`<00<koo`005_oo00<00?oo
ool0>Ooo0P00c?oo000Fool00`00ooooo`0kool30039ool001Koo`8003ooo`800<Ooo`005_oo00<0
0?ooool0@?oo0`00a?oo000Fool00`00ooooo`13ool20032ool001Koo`03003ooooo04Goo`800<3o
o`005_oo00<00?ooool0Aooo0P00__oo000Fool00`00ooooo`19ool2002lool001Koo`03003ooooo
04_oo`800;[oo`005_oo0P00C_oo0P00^?oo000Fool00`00ooooo`1?ool2002fool001Koo`03003o
oooo057oo`800;Coo`005_oo00<00?ooool0Dooo0P00/_oo0003ool20004ool00`00ooooo`04ool0
0`00ooooo`03ool00`00ooooo`1Eool2002`ool000;oo`04003ooooo000:ool00`00ooooo`03ool0
0`00ooooo`1Gool2002^ool000;oo`04003ooooo0007ool50004ool3001Iool2002/ool000;oo`04
003ooooo0007ool01000ooooo`001Ooo00<00?ooool0Fooo00<00?ooool0ZOoo0002ool01000oooo
o`002?oo00<00?oo00001Ooo00<00?ooool0G?oo0P00ZOoo0002ool01000ooooo`002Ooo0P001Ooo
00<00?ooool0G_oo0P00Yooo0003ool2000;ool00`00ooooo`03ool00`00ooooo`1Pool00`00oooo
o`2Tool001Koo`03003ooooo067oo`800:Coo`005_oo0P00I?oo00<00?ooool0XOoo000Fool00`00
ooooo`1Tool2002Qool001Koo`03003ooooo06Koo`03003ooooo09koo`005_oo00<00?ooool0Iooo
0P00W_oo000Fool00`00ooooo`1Yool2002Lool001Koo`03003ooooo06_oo`03003ooooo09Woo`00
5_oo00<00?ooool0K?oo00<00?ooool0V?oo000Fool2001^ool2002Hool001Koo`03003ooooo06oo
o`03003ooooo09Goo`005_oo00<00?ooool0L?oo00<00?ooool0U?oo000Fool00`00ooooo`1aool2
002Dool001Koo`03003ooooo07?oo`03003ooooo097oo`005_oo00<00?ooool0M?oo0P00TOoo000F
ool2001gool00`00ooooo`2>ool001Koo`03003ooooo07Ooo`03003ooooo08goo`005_oo00<00?oo
ool0N?oo0P00SOoo000Fool00`00ooooo`1jool00`00ooooo`2:ool001Koo`03003ooooo07_oo`03
003ooooo08Woo`005_oo00<00?ooool0O?oo00<00?ooool0R?oo000Fool00`00ooooo`1mool00`00
ooooo`27ool001Koo`8007ooo`03003ooooo08Koo`005_oo00<00?ooool0Oooo00<00?ooool0QOoo
000Fool00`00ooooo`20ool00`00ooooo`24ool001Koo`03003ooooo087oo`03003ooooo08?oo`00
0ooo0P001?oo00<00?ooool00_oo0`001Ooo00<00?ooool0P_oo00<00?ooool0P_oo0002ool01000
ooooo`002ooo00<00?ooool00_oo00<00?ooool0Pooo00<00?ooool0POoo0002ool01000ooooo`00
2ooo00<00?ooool00_oo0`00Q?oo00<00?ooool0P?oo0002ool01000ooooo`002?oo0`001Ooo00<0
0?ooool0QOoo00<00?ooool0Oooo0002ool01000ooooo`002?oo00<00?ooool01Ooo00<00?ooool0
Q_oo00<00?ooool0O_oo0002ool01000ooooo`002?oo00<00?ooool01Ooo00<00?ooool0Qooo00<0
0?ooool0OOoo0003ool20009ool40004ool00`00ooooo`28ool00`00ooooo`1lool001Koo`03003o
oooo08Woo`03003ooooo07_oo`005_oo0P00Rooo00<00?ooool0N_oo000Fool00`00ooooo`2;ool0
0`00ooooo`1iool001Koo`03003ooooo08coo`03003ooooo07Soo`005_oo00<00?ooool0SOoo00<0
0?ooool0Mooo000Fool00`00ooooo`2>ool00`00ooooo`1fool001Koo`03003ooooo08ooo`03003o
oooo07Goo`005_oo00<00?ooool0T?oo00<00?ooool0M?oo000Fool2002Bool00`00ooooo`1cool0
01Koo`03003ooooo09;oo`03003ooooo07;oo`005_oo00<00?ooool0T_oo00<00?ooool0L_oo000F
ool00`00ooooo`2Cool00`00ooooo`1aool001Koo`03003ooooo09Coo`03003ooooo073oo`005_oo
00<00?ooool0UOoo00<00?ooool0Kooo000Fool2002Gool00`00ooooo`1^ool001Koo`03003ooooo
09Koo`03003ooooo06koo`005_oo00<00?ooool0Uooo00<00?ooool0KOoo000Fool00`00ooooo`2H
ool00`00ooooo`1/ool001Koo`03003ooooo09Woo`03003ooooo06_oo`005_oo00<00?ooool0VOoo
00<00?ooool0Jooo000Fool00`00ooooo`2Jool00`00ooooo`1Zool001Koo`8009coo`03003ooooo
00_ooa0004koo`005_oo00<00?ooool0W?oo00<00?ooool01Ooo1@004?oo1@00BOoo000Fool00`00
ooooo`2Lool8000Jool50014ool001Koo`03003ooooo09goo`03003ooooo02?oo`<0047oo`000ooo
0P001?oo00<00?ooool00ooo0P001Ooo00<00?ooool0a_oo0`00?_oo0002ool01000ooooo`002?oo
00@00?ooool000Coo`03003ooooo0<Woo`8003coo`000_oo00@00?ooool000Soo`04003ooooo0004
ool3003;ool2000jool000;oo`04003ooooo0008ool30005ool00`00ooooo`3=ool2000hool000;o
o`04003ooooo0008ool00`00ooooo`05ool00`00ooooo`3?ool2000fool000;oo`04003ooooo0008
ool00`00ooooo`05ool00`00ooooo`3Aool2000dool000?oo`8000[oo`<000Coo`03003ooooo0=?o
o`8003;oo`005_oo00<00?ooool0eOoo0P00<?oo000Fool2003Hool00`00ooooo`0]ool001Koo`03
003ooooo0=Soo`03003ooooo02coo`005_oo00<00?ooool0fOoo0P00;?oo000Fool00`00ooooo`3K
ool00`00ooooo`0Yool001Koo`03003ooooo0=coo`03003ooooo02Soo`005_oo00<00?ooool0gOoo
00<00?ooool09ooo000Fool00`00ooooo`3Nool00`00ooooo`0Vool001Koo`800>3oo`03003ooooo
02Goo`005_oo00<00?ooool0h?oo00<00?ooool09?oo000Fool00`00ooooo`3Qool00`00ooooo`0S
ool001Koo`03003ooooo0>;oo`03003ooooo02;oo`005_oo00<00?ooool0hooo00<00?ooool08Ooo
000Fool00`00ooooo`3Sool00`00ooooo`0Qool001Koo`800>Goo`03003ooooo023oo`005_oo00<0
0?ooool0iOoo00<00?ooool07ooo000Fool00`00ooooo`3Vool00`00ooooo`0Nool001Koo`03003o
oooo0>Ooo`03003ooooo01goo`005_oo00<00?ooool0j?oo00<00?ooool07?oo000Fool00`00oooo
o`3Yool00`00ooooo`0Kool001Koo`03003ooooo0>Woo`03003ooooo01_oo`005_oo0P00jooo00<0
0?ooool06_oo000Fool00`00ooooo`3[ool00`00ooooo`0Iool001Koo`03003ooooo0>coo`03003o
oooo01Soo`005_oo00<00?ooool0kOoo00<00?ooool05ooo0003ool20004ool00`00ooooo`03ool0
0`00ooooo`04ool00`00ooooo`3]ool00`00ooooo`0Gool000;oo`04003ooooo0009ool00`00oooo
o`04ool00`00ooooo`3^ool00`00ooooo`0Fool000;oo`04003ooooo000:ool00`00ooooo`03ool3
003^ool00`00ooooo`0Fool000;oo`04003ooooo000:ool00`00ooooo`03ool00`00ooooo`3_ool0
0`00ooooo`0Eool000;oo`04003ooooo000;ool00`00ooooo`02ool00`00ooooo`3_ool00`00oooo
o`0Eool000;oo`04003ooooo0008ool01000ooooo`001?oo00<00?ooool0l?oo00<00?ooool05?oo
0003ool20009ool40004ool00`00ooooo`3`ool00`00ooooo`0Dool001Koo`03003ooooo0?7oo`03
003ooooo01?oo`005_oo0P00looo00<00?ooool04_oo000Fool00`00ooooo`3bool00`00ooooo`0B
ool001Koo`03003ooooo0??oo`03003ooooo017oo`005_oo00<00?ooool0looo00<00?ooool04Ooo
000Fool00`00ooooo`3dool00`00ooooo`0@ool001Koo`03003ooooo0?Coo`03003ooooo013oo`00
5_oo00<00?ooool0mOoo00<00?ooool03ooo000Fool2003fool00`00ooooo`0?ool001Koo`03003o
oooo0?Koo`03003ooooo00koo`005_oo00<00?ooool0m_oo00<00?ooool03_oo000Fool00`00oooo
o`3gool00`00ooooo`0=ool001Koo`03003ooooo0?Ooo`03003ooooo00goo`005_oo00<00?ooool0
n?oo00<00?ooool03?oo000Fool2003iool00`00ooooo`0<ool001Koo`03003ooooo0?Woo`03003o
oooo00_oo`005_oo00<00?ooool0nOoo00<00?ooool02ooo000Fool00`00ooooo`3jool00`00oooo
o`0:ool001Koo`03003ooooo0?[oo`03003ooooo00[oo`005_oo00<00?ooool0nooo00<00?ooool0
2Ooo000Fool00`00ooooo`3kool00`00ooooo`09ool001Koo`800?goo`03003ooooo00Soo`005_oo
00<00?ooool0o?oo00<00?ooool02?oo000Fool00`00ooooo`3mool00`00ooooo`07ool001Koo`03
003ooooo0?goo`03003ooooo00Ooo`000ooo0P001?oo00<00?ooool00ooo0P001Ooo00<00?ooool0
oOoo00<00?ooool01ooo0002ool01000ooooo`002?oo00@00?ooool000Coo`03003ooooo0?koo`03
003ooooo00Koo`000_oo00@00?ooool000Soo`04003ooooo0004ool3003oool8ool000;oo`04003o
oooo0009ool20005ool00`00ooooo`3oool8ool000;oo`04003ooooo0008ool01000ooooo`001?oo
00<00?ooool0oooo2?oo0002ool01000ooooo`002?oo00@00?ooool00?ooo`ooo`000ooo0P002_oo
0P00oooo4?oo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0884228, 0.261369, \
0.00389745, 0.00312361}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[32]="]
}, Open  ]],

Cell["\<\
Examining the mesh points it is clear that far fewer steps were \
taken by the method and that the problematic behavior encountered near the \
discontinuity has been eliminated.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(StepPlot[Join[data1, data2]]\)], "Input",
  CellLabel->"In[33]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0237127 0.952478 0.00456675 0.0101483 [
[.02371 -0.0125 -3 -9 ]
[.02371 -0.0125 3 0 ]
[.21421 -0.0125 -9 -9 ]
[.21421 -0.0125 9 0 ]
[.4047 -0.0125 -9 -9 ]
[.4047 -0.0125 9 0 ]
[.5952 -0.0125 -9 -9 ]
[.5952 -0.0125 9 0 ]
[.78569 -0.0125 -9 -9 ]
[.78569 -0.0125 9 0 ]
[.97619 -0.0125 -3 -9 ]
[.97619 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .00457 -6 -4.5 ]
[-0.0125 .00457 0 4.5 ]
[-0.0125 .10605 -12 -4.5 ]
[-0.0125 .10605 0 4.5 ]
[-0.0125 .20753 -12 -4.5 ]
[-0.0125 .20753 0 4.5 ]
[-0.0125 .30902 -12 -4.5 ]
[-0.0125 .30902 0 4.5 ]
[-0.0125 .4105 -12 -4.5 ]
[-0.0125 .4105 0 4.5 ]
[-0.0125 .51198 -12 -4.5 ]
[-0.0125 .51198 0 4.5 ]
[-0.0125 .61347 -12 -4.5 ]
[-0.0125 .61347 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02371 0 m
.02371 .00625 L
s
[(0)] .02371 -0.0125 0 1 Mshowa
.21421 0 m
.21421 .00625 L
s
[(0.2)] .21421 -0.0125 0 1 Mshowa
.4047 0 m
.4047 .00625 L
s
[(0.4)] .4047 -0.0125 0 1 Mshowa
.5952 0 m
.5952 .00625 L
s
[(0.6)] .5952 -0.0125 0 1 Mshowa
.78569 0 m
.78569 .00625 L
s
[(0.8)] .78569 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(1)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07134 0 m
.07134 .00375 L
s
.11896 0 m
.11896 .00375 L
s
.16658 0 m
.16658 .00375 L
s
.26183 0 m
.26183 .00375 L
s
.30946 0 m
.30946 .00375 L
s
.35708 0 m
.35708 .00375 L
s
.45233 0 m
.45233 .00375 L
s
.49995 0 m
.49995 .00375 L
s
.54758 0 m
.54758 .00375 L
s
.64282 0 m
.64282 .00375 L
s
.69045 0 m
.69045 .00375 L
s
.73807 0 m
.73807 .00375 L
s
.83332 0 m
.83332 .00375 L
s
.88094 0 m
.88094 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .00457 m
.00625 .00457 L
s
[(0)] -0.0125 .00457 1 0 Mshowa
0 .10605 m
.00625 .10605 L
s
[(10)] -0.0125 .10605 1 0 Mshowa
0 .20753 m
.00625 .20753 L
s
[(20)] -0.0125 .20753 1 0 Mshowa
0 .30902 m
.00625 .30902 L
s
[(30)] -0.0125 .30902 1 0 Mshowa
0 .4105 m
.00625 .4105 L
s
[(40)] -0.0125 .4105 1 0 Mshowa
0 .51198 m
.00625 .51198 L
s
[(50)] -0.0125 .51198 1 0 Mshowa
0 .61347 m
.00625 .61347 L
s
[(60)] -0.0125 .61347 1 0 Mshowa
.125 Mabswid
0 .02486 m
.00375 .02486 L
s
0 .04516 m
.00375 .04516 L
s
0 .06546 m
.00375 .06546 L
s
0 .08575 m
.00375 .08575 L
s
0 .12635 m
.00375 .12635 L
s
0 .14664 m
.00375 .14664 L
s
0 .16694 m
.00375 .16694 L
s
0 .18724 m
.00375 .18724 L
s
0 .22783 m
.00375 .22783 L
s
0 .24813 m
.00375 .24813 L
s
0 .26842 m
.00375 .26842 L
s
0 .28872 m
.00375 .28872 L
s
0 .32931 m
.00375 .32931 L
s
0 .34961 m
.00375 .34961 L
s
0 .36991 m
.00375 .36991 L
s
0 .3902 m
.00375 .3902 L
s
0 .4308 m
.00375 .4308 L
s
0 .45109 m
.00375 .45109 L
s
0 .47139 m
.00375 .47139 L
s
0 .49169 m
.00375 .49169 L
s
0 .53228 m
.00375 .53228 L
s
0 .55258 m
.00375 .55258 L
s
0 .57287 m
.00375 .57287 L
s
0 .59317 m
.00375 .59317 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.02371 .61178 m
.02371 .61803 L
s
.21421 .61178 m
.21421 .61803 L
s
.4047 .61178 m
.4047 .61803 L
s
.5952 .61178 m
.5952 .61803 L
s
.78569 .61178 m
.78569 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.07134 .61428 m
.07134 .61803 L
s
.11896 .61428 m
.11896 .61803 L
s
.16658 .61428 m
.16658 .61803 L
s
.26183 .61428 m
.26183 .61803 L
s
.30946 .61428 m
.30946 .61803 L
s
.35708 .61428 m
.35708 .61803 L
s
.45233 .61428 m
.45233 .61803 L
s
.49995 .61428 m
.49995 .61803 L
s
.54758 .61428 m
.54758 .61803 L
s
.64282 .61428 m
.64282 .61803 L
s
.69045 .61428 m
.69045 .61803 L
s
.73807 .61428 m
.73807 .61803 L
s
.83332 .61428 m
.83332 .61803 L
s
.88094 .61428 m
.88094 .61803 L
s
.92857 .61428 m
.92857 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .00457 m
1 .00457 L
s
.99375 .10605 m
1 .10605 L
s
.99375 .20753 m
1 .20753 L
s
.99375 .30902 m
1 .30902 L
s
.99375 .4105 m
1 .4105 L
s
.99375 .51198 m
1 .51198 L
s
.99375 .61347 m
1 .61347 L
s
.125 Mabswid
.99625 .02486 m
1 .02486 L
s
.99625 .04516 m
1 .04516 L
s
.99625 .06546 m
1 .06546 L
s
.99625 .08575 m
1 .08575 L
s
.99625 .12635 m
1 .12635 L
s
.99625 .14664 m
1 .14664 L
s
.99625 .16694 m
1 .16694 L
s
.99625 .18724 m
1 .18724 L
s
.99625 .22783 m
1 .22783 L
s
.99625 .24813 m
1 .24813 L
s
.99625 .26842 m
1 .26842 L
s
.99625 .28872 m
1 .28872 L
s
.99625 .32931 m
1 .32931 L
s
.99625 .34961 m
1 .34961 L
s
.99625 .36991 m
1 .36991 L
s
.99625 .3902 m
1 .3902 L
s
.99625 .4308 m
1 .4308 L
s
.99625 .45109 m
1 .45109 L
s
.99625 .47139 m
1 .47139 L
s
.99625 .49169 m
1 .49169 L
s
.99625 .53228 m
1 .53228 L
s
.99625 .55258 m
1 .55258 L
s
.99625 .57287 m
1 .57287 L
s
.99625 .59317 m
1 .59317 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.008 w
.64461 .40035 Mdot
.65536 .4105 Mdot
.63386 .38006 Mdot
.63924 .3902 Mdot
.6876 .44095 Mdot
.69835 .45109 Mdot
.66611 .42065 Mdot
.67685 .4308 Mdot
.6177 .32931 Mdot
.6195 .33946 Mdot
.60148 .30902 Mdot
.6176 .31917 Mdot
.62311 .35976 Mdot
.62849 .36991 Mdot
.62131 .34961 Mdot
.88328 .55258 Mdot
.90306 .56273 Mdot
.84373 .53228 Mdot
.8635 .54243 Mdot
.9594 .59317 Mdot
.97619 .60332 Mdot
.92283 .57287 Mdot
.94261 .58302 Mdot
.75126 .48154 Mdot
.7689 .49169 Mdot
.71599 .46124 Mdot
.73362 .47139 Mdot
.80417 .51198 Mdot
.82395 .52213 Mdot
.78653 .50184 Mdot
.08426 .10605 Mdot
.10159 .1162 Mdot
.05827 .08575 Mdot
.06693 .0959 Mdot
.15359 .14664 Mdot
.1838 .15679 Mdot
.11893 .12635 Mdot
.13626 .1365 Mdot
.02669 .03501 Mdot
.02948 .04516 Mdot
.02381 .01472 Mdot
.02391 .02486 Mdot
.04093 .06546 Mdot
.0496 .07561 Mdot
.03227 .05531 Mdot
.48073 .25828 Mdot
.50573 .26842 Mdot
.42552 .23798 Mdot
.45573 .24813 Mdot
.55573 .28872 Mdot
.58073 .29887 Mdot
.53073 .27857 Mdot
.27445 .18724 Mdot
.30466 .19739 Mdot
.21402 .16694 Mdot
.24423 .17709 Mdot
.36509 .21768 Mdot
.3953 .22783 Mdot
.33487 .20753 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[33]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool001Woo`8002[oo`8000Coo`03003ooooo00;oo`@002?oo`8000Coo`03003ooooo00Coo`03003o
oooo027oo`8000Coo`03003ooooo00?oo`8002Coo`8000Coo`03003ooooo00?oo`8002Woo`@000_o
o`006?oo00@00?ooool002Soo`04003ooooo0008ool00`00ooooo`0Sool01000ooooo`002_oo00<0
0?ooool08?oo00@00?ooool000Soo`04003ooooo000Rool01000ooooo`002?oo00@00?ooool002[o
o`03003ooooo00[oo`006?oo00@00?ooool002Soo`04003ooooo0009ool00`00ooooo`0Rool01000
ooooo`001ooo1@008Ooo00@00?ooool000Soo`04003ooooo000Rool01000ooooo`002?oo00@00?oo
ool002[oo`03003ooooo00[oo`006?oo00@00?ooool002Soo`04003ooooo000:ool00`00ooooo`0Q
ool01000ooooo`001ooo00@00?ooool002;oo`04003ooooo0008ool3000Sool01000ooooo`002Ooo
0P00:ooo00<00?ooool02_oo000Hool01000ooooo`00:?oo00@00?ooool000_oo`03003ooooo023o
o`04003ooooo0008ool00`00ool0000Rool01000ooooo`002?oo00<00?ooool08ooo00@00?ooool0
00Soo`04003ooooo000Zool00`00ooooo`0:ool001Soo`04003ooooo000Xool01000ooooo`002?oo
00@00?ooool002;oo`04003ooooo0009ool2000Rool01000ooooo`002?oo00<00?ooool08ooo00@0
0?ooool000Soo`04003ooooo000Xool3000<ool001Woo`8002[oo`8000[oo`8002Coo`8000_oo`03
003ooooo027oo`8000[oo`<002?oo`8000[oo`8002_oo`03003ooooo00[oo`00oooo8Ooo003ooolQ
ool000goo`800?oooa;oo`003?oo00@00?ooool000Coool000L000Koo`003?oo00@00?ooool000Co
o`<000?oo`03003ooooo00[oo`03003ooooo00Woo`03003ooooo00[oo`03003ooooo00Woo`03003o
oooo00Woo`03003ooooo00[oo`03003ooooo00Woo`03003ooooo00[oo`03003ooooo00Woo`03003o
oooo00[oo`03003ooooo00Woo`03003ooooo00Woo`03003ooooo00[oo`03003ooooo00Woo`03003o
oooo00[oo`03003ooooo00Woo`03003ooooo00[oo`03003ooooo00Woo`03003ooooo00Woo`03003o
oooo00[oo`04003oooooool30006ool000coo`04003ooooo0004ool00`00ooooo`3oool3ool00`00
ooooo`04ool000coo`04003ooooo0004ool00`00ooooo`03ool2003mool00`00ooooo`04ool000co
o`04003ooooo0004ool00`00ooooo`04ool00`00ooooo`3kool00`00ooooo`04ool000goo`8000Go
o`03003ooooo0?ooo`?oo`03003ooooo00Coo`005?oo0P001?oo0P00o?oo0P001_oo000Dool00`00
ooooo`04ool00`00ooooo`3kool00`00ooooo`04ool001Coo`03003ooooo00?oo`800?goo`03003o
oooo00Coo`005?oo00<00?ooool01?oo00<00?ooool0nooo00<00?ooool01?oo000Dool00`00oooo
o`3oool3ool00`00ooooo`04ool001Coo`03003ooooo00Coo`800?coo`03003ooooo00Coo`005?oo
0P001_oo00<00?ooool0nOoo0P001_oo000Dool00`00ooooo`3oool3ool00`00ooooo`04ool001Co
o`03003ooooo00Goo`800?_oo`03003ooooo00Coo`005?oo00<00?ooool01_oo00<00?ooool0nOoo
00<00?ooool01?oo000Dool00`00ooooo`07ool2003iool00`00ooooo`04ool001Coo`8000Woo`03
003ooooo0?Koo`8000Koo`005?oo00<00?ooool0oooo0ooo00<00?ooool01?oo000Dool00`00oooo
o`09ool2003gool00`00ooooo`04ool001Coo`03003ooooo00[oo`03003ooooo0?Goo`03003ooooo
00Coo`005?oo00<00?ooool0oooo0ooo00<00?ooool01?oo000Dool2000=ool2003cool20006ool0
01Coo`03003ooooo00goo`03003ooooo0?;oo`03003ooooo00Coo`005?oo00<00?ooool03_oo0P00
l_oo00<00?ooool01?oo000Dool00`00ooooo`0?ool00`00ooooo`3`ool00`00ooooo`04ool000Ko
o`@000?oo`8000Goo`03003ooooo0?ooo`?oo`03003ooooo00Coo`002?oo00D00?ooooooo`0000;o
o`03003ooooo00;oo`03003ooooo01?oo`800>goo`03003ooooo00Coo`002?oo00D00?ooooooo`00
00;oo`03003ooooo00;oo`<001Coo`03003ooooo0>Woo`<000Koo`002?oo00D00?ooooooo`0000;o
o`03003ooooo00;oo`03003ooooo0?ooo`?oo`03003ooooo00Coo`002?oo00D00?ooooooo`0000;o
o`03003ooooo00;oo`03003ooooo01Ooo`800>Woo`03003ooooo00Coo`001_oo0`000ooo00@00?oo
ool000Coo`03003ooooo01Soo`03003ooooo0>Ooo`03003ooooo00Coo`002?oo00<00?ooool00_oo
0P001Ooo00<00?ooool07?oo0P00i?oo00<00?ooool01?oo000Dool2000Nool00`00ooooo`3Qool2
0006ool001Coo`03003ooooo0?ooo`?oo`03003ooooo00Coo`005?oo00<00?ooool08?oo0P00h?oo
00<00?ooool01?oo000Dool00`00ooooo`0Qool00`00ooooo`3Nool00`00ooooo`04ool001Coo`03
003ooooo0?ooo`?oo`03003ooooo00Coo`005?oo0P009_oo0P00f_oo0P001_oo000Dool00`00oooo
o`0Vool00`00ooooo`3Iool00`00ooooo`04ool001Coo`03003ooooo02goo`800=?oo`03003ooooo
00Coo`005?oo00<00?ooool0;_oo00<00?ooool0dOoo00<00?ooool01?oo000Dool00`00ooooo`3o
ool3ool00`00ooooo`04ool001Coo`8003Goo`800<_oo`8000Koo`005?oo00<00?ooool0=Ooo00<0
0?ooool0b_oo00<00?ooool01?oo000Dool00`00ooooo`3oool3ool00`00ooooo`04ool001Coo`03
003ooooo03coo`800<Coo`03003ooooo00Coo`005?oo00<00?ooool0?Ooo00<00?ooool0`_oo00<0
0?ooool01?oo000Dool00`00ooooo`14ool2002lool00`00ooooo`04ool001Coo`8004Koo`03003o
oooo0;Woo`8000Koo`005?oo00<00?ooool0oooo0ooo00<00?ooool01?oo000Dool00`00ooooo`1<
ool2002dool00`00ooooo`04ool000Koo`@000?oo`8000Goo`03003ooooo04goo`03003ooooo0;;o
o`03003ooooo00Coo`001_oo00<00?ooool00ooo00@00?ooool000Coo`03003ooooo0?ooo`?oo`03
003ooooo00Coo`001ooo00<00?ooool00_oo00@00?ooool000Coo`<005Coo`800:[oo`<000Koo`00
2?oo00D00?ooooooo`0000;oo`03003ooooo00;oo`03003ooooo05Goo`03003ooooo0:[oo`03003o
oooo00Coo`002Ooo00@00?ooool000;oo`03003ooooo00;oo`03003ooooo05coo`800:Coo`03003o
oooo00Coo`001_oo00@00?ooool000;oo`04003ooooo0004ool00`00ooooo`1Mool00`00ooooo`2R
ool00`00ooooo`04ool000Ooo`8000Coo`8000Goo`03003ooooo0?ooo`?oo`03003ooooo00Coo`00
5?oo0P00IOoo0P00Vooo0P001_oo000Dool00`00ooooo`1Uool00`00ooooo`2Jool00`00ooooo`04
ool001Coo`03003ooooo06coo`8009Coo`03003ooooo00Coo`005?oo00<00?ooool0KOoo00<00?oo
ool0T_oo00<00?ooool01?oo000Dool00`00ooooo`3oool3ool00`00ooooo`04ool001Coo`03003o
oooo07Coo`8008coo`03003ooooo00Coo`005?oo0P00M_oo00<00?ooool0ROoo0P001_oo000Dool0
0`00ooooo`3oool3ool00`00ooooo`04ool001Coo`03003ooooo07[oo`8008Koo`03003ooooo00Co
o`005?oo00<00?ooool0Nooo00<00?ooool0Q?oo00<00?ooool01?oo000Dool00`00ooooo`21ool2
001oool00`00ooooo`04ool001Coo`8008?oo`03003ooooo07coo`8000Koo`005?oo00<00?ooool0
oooo0ooo00<00?ooool01?oo000Dool00`00ooooo`27ool2001iool00`00ooooo`04ool001Coo`03
003ooooo08Soo`03003ooooo07Ooo`03003ooooo00Coo`005?oo00<00?ooool0oooo0ooo00<00?oo
ool01?oo000Dool2002?ool2001aool20006ool001Coo`03003ooooo08ooo`03003ooooo073oo`03
003ooooo00Coo`005?oo00<00?ooool0U?oo0P00K?oo00<00?ooool01?oo000Dool00`00ooooo`2E
ool00`00ooooo`1Zool00`00ooooo`04ool000Ooo`8000Coo`8000Goo`03003ooooo0?ooo`?oo`03
003ooooo00Coo`001_oo00@00?ooool000;oo`04003ooooo0004ool00`00ooooo`2Jool2001Vool0
0`00ooooo`04ool000Woo`04003ooooo0002ool00`00ooooo`02ool3002Kool00`00ooooo`1Rool3
0006ool000Ooo`8000?oo`04003ooooo0004ool00`00ooooo`3oool3ool00`00ooooo`04ool000Wo
o`04003ooooo0002ool00`00ooooo`02ool00`00ooooo`2Nool2001Rool00`00ooooo`04ool000Ko
o`04003ooooo0002ool01000ooooo`001?oo00<00?ooool0Wooo00<00?ooool0H?oo00<00?ooool0
1?oo0007ool20004ool20005ool00`00ooooo`2Nool2001Rool00`00ooooo`04ool001Coo`800:3o
o`03003ooooo05ooo`8000Koo`005?oo00<00?ooool0oooo0ooo00<00?ooool01?oo000Dool00`00
ooooo`2Nool2001Rool00`00ooooo`04ool001Coo`03003ooooo09ooo`03003ooooo063oo`03003o
oooo00Coo`005?oo00<00?ooool0oooo0ooo00<00?ooool01?oo000Dool2002Pool2001Pool20006
ool001Coo`03003ooooo0:3oo`03003ooooo05ooo`03003ooooo00Coo`005?oo00<00?ooool0Wooo
0P00HOoo00<00?ooool01?oo000Dool00`00ooooo`2Pool00`00ooooo`1Oool00`00ooooo`04ool0
01Coo`03003ooooo0?ooo`?oo`03003ooooo00Coo`005?oo0P00X_oo0P00G_oo0P001_oo000Dool0
0`00ooooo`2Rool00`00ooooo`1Mool00`00ooooo`04ool001Coo`03003ooooo0?ooo`?oo`03003o
oooo00Coo`005?oo00<00?ooool0X_oo0P00G_oo00<00?ooool01?oo000Dool00`00ooooo`2Sool0
0`00ooooo`1Lool00`00ooooo`04ool001Coo`03003ooooo0:?oo`8005goo`03003ooooo00Coo`00
5?oo0P00YOoo00<00?ooool0F_oo0P001_oo000Dool00`00ooooo`3oool3ool00`00ooooo`04ool0
01Coo`03003ooooo0:Goo`8005_oo`03003ooooo00Coo`002?oo00<00?ooool00_oo0P001Ooo00<0
0?ooool0Y_oo00<00?ooool0FOoo00<00?ooool01?oo0008ool01@00oooooooo00000_oo00<00?oo
ool00_oo00<00?ooool0oooo0ooo00<00?ooool01?oo0005ool50002ool01000ooooo`001?oo0`00
Z?oo0P00E_oo0`001_oo0005ool01000ooooo`000ooo00@00?ooool000Coo`03003ooooo0:Woo`03
003ooooo05Koo`03003ooooo00Coo`001_oo00<00?oo00000ooo00@00?ooool000Coo`03003ooooo
0:[oo`8005Koo`03003ooooo00Coo`001ooo0P000ooo00@00?ooool000Coo`03003ooooo0:_oo`03
003ooooo05Coo`03003ooooo00Coo`002?oo00<00?ooool00_oo0P001Ooo00<00?ooool0oooo0ooo
00<00?ooool01?oo000Dool2002^ool2001Bool20006ool001Coo`03003ooooo0:koo`03003ooooo
057oo`03003ooooo00Coo`005?oo00<00?ooool0/?oo0P00D?oo00<00?ooool01?oo000Dool00`00
ooooo`2aool00`00ooooo`1>ool00`00ooooo`04ool001Coo`03003ooooo0?ooo`?oo`03003ooooo
00Coo`005?oo00<00?ooool0/ooo0P00COoo00<00?ooool01?oo000Dool2002eool00`00ooooo`1:
ool20006ool001Coo`03003ooooo0?ooo`?oo`03003ooooo00Coo`005?oo00<00?ooool0]ooo0P00
BOoo00<00?ooool01?oo000Dool00`00ooooo`2hool00`00ooooo`17ool00`00ooooo`04ool001Co
o`03003ooooo0;coo`8004Coo`03003ooooo00Coo`005?oo0P00__oo00<00?ooool0@Ooo0P001_oo
000Dool00`00ooooo`3oool3ool00`00ooooo`04ool001Coo`03003ooooo0<7oo`8003ooo`03003o
oooo00Coo`005?oo00<00?ooool0`_oo00<00?ooool0?Ooo00<00?ooool01?oo000Dool00`00oooo
o`3oool3ool00`00ooooo`04ool001Coo`800<Koo`8003[oo`8000Koo`005?oo00<00?ooool0a_oo
00<00?ooool0>Ooo00<00?ooool01?oo000Dool00`00ooooo`3:ool2000fool00`00ooooo`04ool0
00Koo`<000Coo`8000Goo`03003ooooo0<_oo`03003ooooo03Coo`03003ooooo00Coo`002Ooo00@0
0?ooool000;oo`03003ooooo00;oo`03003ooooo0?ooo`?oo`03003ooooo00Coo`002Ooo00@00?oo
ool000;oo`03003ooooo00;oo`<00<koo`80033oo`<000Koo`001_oo0`000ooo00@00?ooool000Co
o`03003ooooo0<ooo`03003ooooo033oo`03003ooooo00Coo`001_oo00<00?ooool00ooo00@00?oo
ool000Coo`03003ooooo0?ooo`?oo`03003ooooo00Coo`001_oo00<00?ooool00ooo00@00?ooool0
00Coo`03003ooooo0=Coo`8002coo`03003ooooo00Coo`001_oo10000ooo0P001Ooo00<00?ooool0
eOoo00<00?ooool0:_oo00<00?ooool01?oo000Dool00`00ooooo`3Iool2000Wool00`00ooooo`04
ool001Coo`800=_oo`03003ooooo02Coo`8000Koo`005?oo00<00?ooool0oooo0ooo00<00?ooool0
1?oo000Dool00`00ooooo`3Nool2000Rool00`00ooooo`04ool001Coo`03003ooooo0=ooo`03003o
oooo023oo`03003ooooo00Coo`005?oo00<00?ooool0oooo0ooo00<00?ooool01?oo000Dool2003T
ool2000Lool20006ool001Coo`03003ooooo0>Coo`03003ooooo01_oo`03003ooooo00Coo`005?oo
00<00?ooool0j?oo0P006?oo00<00?ooool01?oo000Dool00`00ooooo`3Yool00`00ooooo`0Fool0
0`00ooooo`04ool001Coo`03003ooooo0?ooo`?oo`03003ooooo00Coo`005?oo0P00k_oo0P004_oo
0P001_oo000Dool00`00ooooo`3^ool00`00ooooo`0Aool00`00ooooo`04ool001Coo`03003ooooo
0?ooo`?oo`03003ooooo00Coo`005?oo00<00?ooool0looo0P003Ooo00<00?ooool01?oo000Dool0
0`00ooooo`3dool00`00ooooo`0;ool00`00ooooo`04ool001Coo`03003ooooo0?Ooo`8000Woo`03
003ooooo00Coo`005?oo0P00nOoo00<00?ooool01_oo0P001_oo000Dool00`00ooooo`3oool3ool0
0`00ooooo`04ool001Coo`03003ooooo0?_oo`8000Goo`03003ooooo00Coo`001ooo0P001?oo0P00
1Ooo00<00?ooool0o?oo00<00?ooool00ooo00<00?ooool01?oo0006ool01000ooooo`000_oo00@0
0?ooool000Coo`03003ooooo0?ooo`?oo`03003ooooo00Coo`001_oo00@00?ooool000;oo`04003o
oooo0004ool30003ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`09
ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`09
ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`09
ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`0:ool00`00ooooo`09ool00`00ooooo`09
ool00`00ooooo`0:ool01000oooooooo0`001_oo0006ool30003ool01000ooooo`001?ooo`001`00
1_oo0006ool00`00ooooo`03ool01000ooooo`00oooo4Ooo0006ool00`00ooooo`03ool01000oooo
o`00oooo4Ooo0007ool30003ool2003ooolBool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.107423, -5.07873, \
0.00402116, 0.37741}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[33]="]
}, Open  ]],

Cell[TextData[{
  "The value of the discontinuity is given as 0.6234 in [",
  ButtonBox["HNW93",
    ButtonData:>{"References.nb", "HNW93"},
    ButtonStyle->"Hyperlink"],
  "], which coincides with the value found by the ",
  StyleBox["EventLocator", "MR"],
  " method."
}], "Text"],

Cell["\<\
In this example it is possible to analytically solve the system and \
use a numerical method to check the value.\
\>", "Text"],

Cell["\<\
The solution of the system up to the discontinuity can be \
represented in terms of Bessel functions.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(dsol\  = \ FullSimplify[First[DSolve[system1, y[t], t]]]\)], "Input",
  CellLabel->"In[34]:="],

Cell[BoxData[
    \({y[t] \[Rule] \(\@2\ t\ \((3\ BesselJ[\(-\(3\/4\)\), t\^2\/\@2]\ \
Gamma[1\/4] + 10\ 2\^\(1/4\)\ BesselJ[3\/4, t\^2\/\@2]\ \
Gamma[3\/4])\)\)\/\(\(-6\)\ BesselJ[1\/4, t\^2\/\@2]\ Gamma[1\/4] + 20\ \
2\^\(1/4\)\ BesselJ[\(-\(1\/4\)\), t\^2\/\@2]\ Gamma[3\/4]\)}\)], "Output",
  CellLabel->"Out[34]="]
}, Open  ]],

Cell["\<\
Substituting in the solution into the switching function, a local \
minimization confirms the value of the discontinuity.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(FindRoot[event\  /. \ dsol, {t, 3/5}]\)], "Input",
  CellLabel->"In[35]:="],

Cell[BoxData[
    \({t \[Rule] 0.6234179814117705`}\)], "Output",
  CellLabel->"Out[35]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Avoiding Wraparound in PDEs", "Subsection"],

Cell["\<\
Many evolution equations model behavior on a spatial domain which is infinite \
or sufficiently large to make it impractical to discretize the entire domain \
without using specialized discretization methods.  In practice, it is often \
the case that it is possible to use a smaller computational domain for as \
long as the solution of interest remains localized. \
\>", "Text"],

Cell[TextData[{
  "In situations where the boundaries of the computational domain are imposed \
by practical considerations rather than the actual model being studied, it is \
possible to pick boundary conditions appropriately.  Using a  ",
  ButtonBox["pseudospectral",
    ButtonData:>{"PDE.nb", "b:7.1.2"},
    ButtonStyle->"Hyperlink"],
  "  method with periodic boundary conditions can make it possible to \
increase the extent of the computational domain because of the superb \
resolution of the periodic pseudospectral approximation.  The drawback of \
periodic boundary conditions is that signals which propagate past the \
boundary persist on the other side of the domain, affecting the solution \
through wraparound.  It is possible to use an absorbing layer near the \
boundary to minimize these effects, but it is not always possible to \
completely eliminate them."
}], "Text"],

Cell["\<\
The sine-Gordon equation turns up in differential geometry and relativistic \
field theory. This example integrates the equation starting with a localized \
initial condition which spreads out.  The periodic pseudospectral method is \
used for the integration.  Since no absorbing layer has been instituted near \
the boundaries, it is most appropriate to stop the integration once \
wraparound becomes significant.  This condition is easily detected with event \
location using the EventLocator method.\
\>", "Text"],

Cell["\<\
The integration is stopped when the size of the solution at the \
periodic wraparound point crosses a threshold of 0.01, beyond which, the form \
of the wave would be affected by periodicity.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Timing[
      sgsol\  = \ 
        First[NDSolve[{D[u[t, x], t, t]\  \[Equal] \ 
                D[u[t, x], x, x]\  - \ Sin[u[t, x]], \ 
              u[0, x]\  \[Equal] \ \[ExponentialE]\^\(-\((x - 5)\)\^2\) + \
\[ExponentialE]\^\(\(-\((x + 5)\)\^2\)/2\), \ \(\(Derivative[1, 0]\)[u]\)[0, 
                  x]\  \[Equal] \ 0, \ 
              u[t, \(-50\)]\  \[Equal] \ u[t, 50]}, \ 
            u, \ {t, 0, 1000}, \ {x, \(-50\), 50}, \ 
            Method \[Rule] {"\<MethodOfLines\>", \
\[IndentingNewLine]"\<SpatialDiscretization\>" \[Rule] \
{"\<TensorProductGrid\>", \ "\<DifferenceOrder\>" -> "\<Pseudospectral\>"}, 
                Method \[Rule] {"\<EventLocator\>", \ "\<Event\>" \
\[RuleDelayed] \ Abs[u[t, \(-50\)]]\  - \ 
                        0.01, \ "\<EventLocationMethod\>" -> \
"\<StepBegin\>"}}]]]\)], "Input",
  CellLabel->"In[7]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(0.9510000000000001`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{"u", "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 
                    45.81888712328008`}, {"...", \(-50.`\), 50.`, "..."}}, 
                "<>"]\),
              False,
              Editable->False]}], "}"}]}], "}"}]], "Output",
  CellLabel->"Out[7]="]
}, Open  ]],

Cell[TextData[{
  "This extracts the ending time from the ",
  StyleBox["InterpolatingFunction", "MR"],
  " object and makes a plot of the computed solution.  You can see that the \
integration has been stopped just as the first waves begin to reach the \
boundary."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(end\  = \ \(InterpolatingFunctionDomain[
            u\  /. \ sgsol]\)[\([1, \(-1\)]\)];\)\), "\[IndentingNewLine]", 
    \(DensityPlot[
      u[t, x]\  /. \ sgsol, {x, \(-50\), 50}, \ \ {t, 0, \ end}, \ 
      Mesh \[Rule] False, \ PlotPoints \[Rule] 100]\)}], "Input",
  CellLabel->"In[8]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% DensityGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.00961538 0.0192308 0.0209856 [
[.11538 -0.0125 -9 -9 ]
[.11538 -0.0125 9 0 ]
[.30769 -0.0125 -9 -9 ]
[.30769 -0.0125 9 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.69231 -0.0125 -6 -9 ]
[.69231 -0.0125 6 0 ]
[.88462 -0.0125 -6 -9 ]
[.88462 -0.0125 6 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01923 -6 -4.5 ]
[-0.0125 .01923 0 4.5 ]
[-0.0125 .22909 -12 -4.5 ]
[-0.0125 .22909 0 4.5 ]
[-0.0125 .43894 -12 -4.5 ]
[-0.0125 .43894 0 4.5 ]
[-0.0125 .6488 -12 -4.5 ]
[-0.0125 .6488 0 4.5 ]
[-0.0125 .85866 -12 -4.5 ]
[-0.0125 .85866 0 4.5 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.11538 0 m
.11538 .00625 L
s
[(-40)] .11538 -0.0125 0 1 Mshowa
.30769 0 m
.30769 .00625 L
s
[(-20)] .30769 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(0)] .5 -0.0125 0 1 Mshowa
.69231 0 m
.69231 .00625 L
s
[(20)] .69231 -0.0125 0 1 Mshowa
.88462 0 m
.88462 .00625 L
s
[(40)] .88462 -0.0125 0 1 Mshowa
.125 Mabswid
.16346 0 m
.16346 .00375 L
s
.21154 0 m
.21154 .00375 L
s
.25962 0 m
.25962 .00375 L
s
.35577 0 m
.35577 .00375 L
s
.40385 0 m
.40385 .00375 L
s
.45192 0 m
.45192 .00375 L
s
.54808 0 m
.54808 .00375 L
s
.59615 0 m
.59615 .00375 L
s
.64423 0 m
.64423 .00375 L
s
.74038 0 m
.74038 .00375 L
s
.78846 0 m
.78846 .00375 L
s
.83654 0 m
.83654 .00375 L
s
.06731 0 m
.06731 .00375 L
s
.01923 0 m
.01923 .00375 L
s
.93269 0 m
.93269 .00375 L
s
.98077 0 m
.98077 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01923 m
.00625 .01923 L
s
[(0)] -0.0125 .01923 1 0 Mshowa
0 .22909 m
.00625 .22909 L
s
[(10)] -0.0125 .22909 1 0 Mshowa
0 .43894 m
.00625 .43894 L
s
[(20)] -0.0125 .43894 1 0 Mshowa
0 .6488 m
.00625 .6488 L
s
[(30)] -0.0125 .6488 1 0 Mshowa
0 .85866 m
.00625 .85866 L
s
[(40)] -0.0125 .85866 1 0 Mshowa
.125 Mabswid
0 .0612 m
.00375 .0612 L
s
0 .10317 m
.00375 .10317 L
s
0 .14514 m
.00375 .14514 L
s
0 .18712 m
.00375 .18712 L
s
0 .27106 m
.00375 .27106 L
s
0 .31303 m
.00375 .31303 L
s
0 .355 m
.00375 .355 L
s
0 .39697 m
.00375 .39697 L
s
0 .48091 m
.00375 .48091 L
s
0 .52289 m
.00375 .52289 L
s
0 .56486 m
.00375 .56486 L
s
0 .60683 m
.00375 .60683 L
s
0 .69077 m
.00375 .69077 L
s
0 .73274 m
.00375 .73274 L
s
0 .77471 m
.00375 .77471 L
s
0 .81668 m
.00375 .81668 L
s
0 .90063 m
.00375 .90063 L
s
0 .9426 m
.00375 .9426 L
s
0 .98457 m
.00375 .98457 L
s
.25 Mabswid
0 0 m
0 1 L
s
.11538 .99375 m
.11538 1 L
s
.30769 .99375 m
.30769 1 L
s
.5 .99375 m
.5 1 L
s
.69231 .99375 m
.69231 1 L
s
.88462 .99375 m
.88462 1 L
s
.125 Mabswid
.16346 .99625 m
.16346 1 L
s
.21154 .99625 m
.21154 1 L
s
.25962 .99625 m
.25962 1 L
s
.35577 .99625 m
.35577 1 L
s
.40385 .99625 m
.40385 1 L
s
.45192 .99625 m
.45192 1 L
s
.54808 .99625 m
.54808 1 L
s
.59615 .99625 m
.59615 1 L
s
.64423 .99625 m
.64423 1 L
s
.74038 .99625 m
.74038 1 L
s
.78846 .99625 m
.78846 1 L
s
.83654 .99625 m
.83654 1 L
s
.06731 .99625 m
.06731 1 L
s
.01923 .99625 m
.01923 1 L
s
.93269 .99625 m
.93269 1 L
s
.98077 .99625 m
.98077 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .01923 m
1 .01923 L
s
.99375 .22909 m
1 .22909 L
s
.99375 .43894 m
1 .43894 L
s
.99375 .6488 m
1 .6488 L
s
.99375 .85866 m
1 .85866 L
s
.125 Mabswid
.99625 .0612 m
1 .0612 L
s
.99625 .10317 m
1 .10317 L
s
.99625 .14514 m
1 .14514 L
s
.99625 .18712 m
1 .18712 L
s
.99625 .27106 m
1 .27106 L
s
.99625 .31303 m
1 .31303 L
s
.99625 .355 m
1 .355 L
s
.99625 .39697 m
1 .39697 L
s
.99625 .48091 m
1 .48091 L
s
.99625 .52289 m
1 .52289 L
s
.99625 .56486 m
1 .56486 L
s
.99625 .60683 m
1 .60683 L
s
.99625 .69077 m
1 .69077 L
s
.99625 .73274 m
1 .73274 L
s
.99625 .77471 m
1 .77471 L
s
.99625 .81668 m
1 .81668 L
s
.99625 .90063 m
1 .90063 L
s
.99625 .9426 m
1 .9426 L
s
.99625 .98457 m
1 .98457 L
s
.25 Mabswid
1 0 m
1 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% Start of gray image
p
.01923 .01923 translate
.96154 .96154 scale
100 string
100 100 8 [100 0 0 100 0 0] {
\tcurrentfile
\t1 index
\treadhexstring
\tpop
}  Mimage
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7F8EFBFFFFFF927F7E7E7E7FB0FFFFA37F7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E8095FFFFFFFF9B807E7E7E82CBFFFFBC817E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E83AAFCFFFFFFB1857F7E7E8FFADDE9EE8A7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7F8CC1CF564BC8C790807E80B1EC263AF2A7807E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E829CC46E00005DC4A1837F8AD67C000093CE867E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7F88AB9D0000000092AE8A81A2C400000008CC9B7F7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E8193A84A0000000039A69689BD5C0000000073B8847E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7F859C83000000000000799FA0AB000003000000B3927F7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E808C973E00001A1D00003099BC5100154C4A0C0063A8827E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7F83937600003F85884900007EB000016B9593630000A28C7F7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7F888C3E0039ABE1E3B4460054560063B6D0CFB154005E9D807E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E818C701121A6FFFFFFFFB3391E0043BDEAF5F5E7B0240797887E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7F8586441583FFFFFFFFFFFFAA0000A3FAFFFFFFFFEE8F005D94807E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
80886E2655E4FFFFFFFFFFFFFD0003D4FFFCEFF0FBFFE34D1490857E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F
83824C37B3FFFFFFF6F5FFFFF76087CEF7DFCACBDCF8FFBA14608E7F7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E80
856F397AFFFFF9CBB2B1CEFEBBFFFFB1B7B09A99ACD0FCFE6D238C837E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F82
805451CFFFF4B18067678A86A2FFFFA4657368667699D0FFD52A648A7F7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F83
704A93FFFBAA6339272A3E08BAFFFFAE28313E3B456094DEFF843089817E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E817F
5C65DAFFB359200500050000C3FFFFB91F001E222433589DF4E43C68877F7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F8172
57A1FBC9601700000000001D97899BA847000A1C191A2B5BB6FF943D86807E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E807E63
73DBE1761A00000005000064470018737E1D09262319162A6CDAEA4A6C857E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F807461
A8E9992A0000061C2B004B7E0000002F99732B3A402E1B14318DF79E4885807E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7E687C
D5BE4700001437574B3DC56F000000008EC475586554371B1344BDEA5570837E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F807669AA
D4710C001C4C76946AD0F459000000006BE6D0878881623B171869E9A251837F7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7E6D82CB
A028001B578CB5B7B6FFE0582213080D4DD0FFC7A4A89169381228A2E65D74827E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F786FA9BF
5303155899C7E4C3FFFFB371798575514A9BFFFFC1C0B99A692F0F4DD9A459827F7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7E7185C086
180D4F9CD1F0ECE5FFFB919CD1F0DFA4656AD3FFE1C7D0C19C6121178ADF6376817E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7973A6AD40
093F94D4F3FDD2FFFF9C84C9FFFFFFEE93547EEFF7C6D0D7C598511337C9A360827F7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7E7487B67315
2A84D0F4F9E1C7FFB15689E7FFFFFFFFC25E3B94E7C6BCD4D9C38B3A1176D76879807E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7A76A29E351B
6BC4F2F5E2A7D8CF373793EAFFFFFFFFDF7C2233A4BFA3BBD5D9BD762129BBA166817E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7687AC651A4D
B0ECF3D9AB80C842003997CFE8F2FAFAE19B3400439E9094BBD5D8AF571267CE6B7A7F7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7B799E91313193
E2F2D5A863826B00004D8D9D9895A0B6C4AD5D00005C807091BBD8D4983422AF9D6B807E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7887A35D256ECF
F1D6A26839780000116276604436426591A88829000E645C6491BEDBCA75195BC66E7C7F
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7C7A99883349B2ED
DCA3692745330000497058280000001C568FA36C02003556446394C5DDB4491FA69A6F80
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E79869C58338AE0E3
AB68320E540000357E743E0400000000246BA6A75200064C3D3D669BCFD9902454BE707D
7F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7C7B95813760C7E8B8
6F36083232001C8DA071310000000000084995C6A73900374A2E3F6CA8D9C95F1F9E9672
807E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7A85955642A1E4C87C
3A15095F01058ECDAE6E36130000000009357AC6E29B2A24563B2A4376B9DEA7304FB772
7D7F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D7C927C3E74D3D79043
1A09465A097AEBE9AC714F444240392E283562ADF2EC8332565A2F2C4A86CBD673229892
757F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7B84905750B1DEA7531E
1525893F5BEDFFE3A17C768391968D775C4B5689DAFFE06D55774F2C2F559DDAB93D4DB0
737E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D7D8F794685D7C06A2617
26729451CFFFFFC5958EA3C1D9E3DBC19A725B6AAAFFFFC16A817C452C3366B7DC852793
8E777F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7C848C595DBCD18635172C
54BD7B9FFFFFE39F8DA3CAF0FFFFFFFAD29F6F5A77C8FFFF9F819D72402B3C7FCFC6494D
AA757E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D7D8C774D91D4A54A1B2952
A4C684FDFFFCA47D8AB3E0FFFFFFFFFFF5C58C5B5081E2FFE090A5A2673C2A4B9FDB942C
908B797F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7C83895B68C1C06823225287
E9A1BEFFFFAA68668BB9E1FCFFFFFFFFFDDBA76C3F4590F5FFB59DBD975F352D64C0CE55
4EA4767E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D8976559BCD8C351E4883CD
E394FDFFB7573E5C8AB1CAD6DCE0E5E9E8D9B883472341A4FADD9CB9C08B562E3887D6A0
328E887A7F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D82875E71C2AE501F3A7CB2FF
A8B6FFCA531C2D5C859CA19D98979EABB9C1B9975E220E49B9E8AEA1CEB68149294EAED3
5F519F777E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D87765CA1C375292E6DACE4E5
84DCDA5F060533627B7C6F5D4F4A51637D98A8A27C3D04065EC4C291B9D0AB753B2D70CE
A9388C857B7F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D81856279C09B3F2559A3CEFD95
94D57903001048696C5A3F25120B102342678A9E94661F000E79BC9492CCC9A1642F3D9C
D467549B787E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E867662A4B762264591CAE6CF60
AD9615000035646F5D3E1D0100000000143A668DA08E520500268F9D76A7D1C1934F295C
C4B03E8B837C7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7D8184657FBC8A35337AC1D7E77068
9F3A000025657D72522F100000000000001D47759DAA8B3F00004A927376BBCDB9803B31
8AD36E5797797E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E847667A5AB532B60AFD7D6AA3682
640000156791907350311C0E060100000617345E8EB4B7852C000C6A7C578BC3CAAD672D
4CB9B4448A817D7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E80826883B67B314797D2CFC247427C
1C000665A5B1997658463E3C3C3A352F2929344F7AABCDC17B1B00347758579FC5C79A4C
2C7AD0735B937A7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E83776BA5A049357BC5D6B880116850
00005CB2D0BE9A7B6B686E787F807B6E5E4F464D6896C8E3C46B100E5A683E6BABC8BF80
3540ADB6498A807D7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E80826B86B06F325CB0D7BA98242C721E
004CB6E9E0BA97858491A3B5C1C5BFB19A80685A5F7DB0E3F2BE5713376F483D7FB2CBAF
602B6CCB775F907B7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E82786FA495434393D0C8945A0065580E
3BACF8FCD6AB92909FB7D0E5F3F9F5E7D0B1907261688DC7FAF8AD452760662F4E8DBBC9
964238A2B74D897F7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F816E87AA663771C1D19E700E2A803931
97FAFFEDB9968E9BB5D1EBFFFFFFFFFFF3D7B58F6F5F6C9BDDFFF2943F4C78492C6299C5
BF752F60C67A638D7B7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E827872A28C4152A7D2B2703F0076723A7C
EDFFFFC493828BA3C0DAEEFCFFFFFFFFFCE9CEAA8362566DA8EFFFDF7A4A74732F3A70A8
CBAA513398B751897E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F807089A35F3F85CAC27F51093B9E5D66D2
FFFFCE8F707288A4BCCED9DEE2E5E7E9E9E3D4BB976F4F4A6DB6FEFFC168688E57294B7E
B8C8893556C17C678A7C7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E8179749F854162B5CC98523209969463AEFF
FFD98C5D5669849CAAB0AFACAAAAAEB6BFC6C7BDA581583A3E71C5FFF7A06A8E8A3B3257
90C6B961308FB655897E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F8072899E5A4896CCAE623D1359C17E8CEFFF
E58D4E3B4A667E8C8D867B7069676C778798A8AFA8916B4026367AD4FFD88681A76F313C
65A6CB9A3E4EBC7D6A887C7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E807A769D7E4471BFC07D3C3325BAB27EC5FFF0
9544232E4C6876766B5A48382E2B2F3B4E657E949F9A815629153488E0F4B27FA4A55234
4578BBC3713187B459887D7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F807489985853A3C9974B352A7DE1919CEFF5A2
42111535576B6D604C3520100501040E1F3754728C99937443150A3B9AE4D79390BE8B43
385092C9A94849B77E6E867D7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E807B789A7A487EC3B064304047DCC586C4EEB24A
0600224C686E644F37200D000000000006193352738F9B906A3204064AACDCB185B1BE6D
403B62ACC8803381B25C887D7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F80758994575DADC1813B39499FF69497D8C05C05
0012446A79725F473120130B050201030812233C5C7D99A3916122000C60B9C58F92CCA6
5C3E417CC1B55245B27D71847D7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E807B7997764D8AC39E502E556BF4C77BB0C4731000
033D6F8886755F4B3C34302E2D2C2926232228354C6B8DA8AE925713001B79BAA27EB2CF
8A5439509BCA8D377BAF5F877D7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7788905766B3B56C33456BBAFC8682B68B250000
3675989E8F7A675B57585D636667635B5147403F485E7DA0BAB9924C0700348EAB8088CF
BB774A3969B7BD5D43AE7D74827D7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F7C7A94735294C08B42346F8BFFBA6394994300002D
77A8B5A9937F75747A86929DA4A5A1988978665953596E8FB4CBC28F3F000552998D6CA7
D6A26B3F4289C7993B77AC62867D7E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7E7F78878D586FB7A85A32588ACBF46B67966309002376
B4CBC2AA93878790A0B2C2CFD8DBD7CEBEAA937C695F657BA0C7DBC88731001D6E926B74
C7C8905C3757AAC36742AA7C77817E7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7F7C7B9272589BB97A3A4289A2FCA1457A7B2900196FBB
DED8BDA1918F99AABFD2E3F0F8FBFAF2E5D2BBA0856F656B86AFD9E8C97A26093D807956
93D3B4834B3A77C2A34074A865857D7E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7E7F79868A5A77B79A4D376EA4D0E04A4E804E091263BBEB
EBCDAB948E96A7BCD0E1EFF8FEFFFFFDF5E8D5BDA1846D636D8EBEE8F1C26A1F1D5D8159
5CB5CBA5713C4A9CC57042A77B79807E7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7F7D7B8F715DA0B26B3853A1B0EE81286A6D281354B4F3FA
DBB192868A9AADC0CFDAE2E6E9EBEDEDECE7DDCCB6997C655D6C95CBF5F3B558213B756D
427AC6BD995B3767BBAB4572A568857D7E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7E7F7A86875C7EB68D444285B7C9C42A3D784D1F45A5F2FFE8
B68D79798799AAB6BDC0C0BEBEBEC1C6CACECFCABDA88D6F58546A9CD8FEEEA1492E5C7B
4D449EC5B48646408DC67943A47A7B807E7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7F7D7C8D7063A4A95E3A67B4B4D660136670383D90E9FFF3BB
876A657082929C9F9D978F898585888F99A5AFB6B6AD9A7E5F4A4968A4E4FFE08A434777
6A3360B3BFAB6D3959B2B14A70A26A847D7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7E7F7A85865F83B380405099C1B9A41039805B407AD8FFFCC183
5B505A6C7C85868075685C524C4A4E5663738495A1A69F8C6F4F3A4069ADEEFFCC744866
814A3184B7BC9A533A7FC48144A2797C7F7E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7F7D7C8C7067A59F55417BC1AFB94309717F5168BFFFFFC9814F
3D45586B75766E604F3F30251E1C1E263243586E84949B9581613F2B386CB9F5F5B2655A
83732D4A9BB8B8803F4EA8B64F709E6D837E7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7F7B84846187AF753F60AAC3A4860042947161A3F3FFD283452B
31475E6C6E6657453323150B050304091321344B647C8E979078532F1F3474C5F7E29662
778F52266BA5BCAA623873C187469F787E7F7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7F7D7C8A706CA6964E4A8EC7A49C2E0C87966B8AD9FFDC8A411D20
3955696F685A4937271A110A060404060B13213349637C90978E714821153580D0F1C780
6D9383313983ACBE9148459EB9546F9B6F827E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7F7B8383648BAA6C4170B6BE8B6D0056AE877BBBF3E3954213112D
506A7572665545382E282422211F1E1C1C1D222B3A506983969D8F6C3C140F3C8FD8E3AA
7587A36225578FB7B6723A67BD8C489E777F7E7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7E7D88716FA68E4A569DC79380231AA4AC7E9CDDE5A3490E05224C
6E80817667594E4846484A4D4F4E4C47413B37363B475B748DA1A59167310A0E489FDACC
8F7BA4993E316C9BC0A1534094BA58709872817E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7F7C8282668DA5644680BDB3725C0172C89687BFDFB1570F00184974
8D92897B6E6562656B737B828687847D73685C514B4C5567809BADAE946127031359AED3
AF7F92B6782C4879ACBE823E5DB8914A9C767F7E7E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7E7D877272A4864962AAC281692231C2BC859FCFBC6917000F45799A
A49D8F8179787D8894A1ADB6BBBDBAB3A8998877675D5A61738DA9BBB7955A1C001E6EB8
C19381AFB052315A88BCAE603D8BBB5C719474817E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7F7C8281698F9F5F4D8EBFA55C531191DD9A86B5BE7E2600073F7DA7B6
B0A09187868D9AA9B9C7D4DDE3E5E3DDD2C4B39E89766863697C99B7C8BF945114023083
B9A87F95C5913A41669EC1914455B3944C9A76807E7E7E7E7E7E7E7E
7E7E7E7E7E7E7E7E7D867375A37F496EB2B86F592B4DDEC38095B6903B0001387DB1C6C2
B09C908E94A1B1C1D1DEE8F0F5F7F6F2EBE0D1BFA9937D6D676E84A4C4D4C590470D0B47
95AF8C7DB4C469394D76B3B76D3C82BB5F729176807E7E7E7E7E7E7E
7E7E7E7E7E7E7E7D81816B909A5B559ABD944A5227AFEA937AA19B540C002F7AB8D4D1BC
A4948E929EADBDCBD7E0E6EAEDEFEFEFECE7DED1C0AB937D6C67708AAFD1DFC7893B0B1B
609E9A768FCEA84E42568EBF9E4C4FAD974F9975807E7E7E7E7E7E7E
7E7E7E7E7E7E7E7D857477A0794C7AB7AC5E503A69F2BE70849B6C21022774BBE0DFC7AA
93898A94A2B0BCC5CACDCECECFCFD1D3D6D7D6D1C8B9A58E7867626F8EB8DDE6C57D300E
31779B7F71AFD281474764A8BD7A3D7ABA62748E787F7E7E7E7E7E7E
7E7E7E7E7E7E7D81806D9095585DA3B8833E5841C8EC82678C7F3B0B1F69B9E7EBD1AD8F
807E86939FA8AEAFADA9A5A19F9FA2A6ADB4BBBFBFBAAF9C866F5E5C6D92C1E7EABE6F28
194B878B6881CEBB64484A7DBAA9554AA899519875807E7E7E7E7E7E
7E7E7E7E7E7E7D8475799E754F85B99D504D4F84FCAF597186561C1C5CB1EAF5D9AF8A74
6F75818C9497948E857C746D69696B717A86929EA8ADACA3927B6453536A96CBEFEAB360
252B658A725FA2D8975947559ABF873F73B864758B797F7E7E7E7E7E
7E7E7E7E7E7D80806F90915766AAB07238645BDAE469537C6E341E4FA4E8FBE2B285685F
64707B82847F75695C50453D38373A404956667687959EA099887057484B689BD4F5E5A3
512943797F596DC6C87A54446DB3B25F46A39A549676807E7E7E7E7E
7E7E7E7E7E7D83767A9C71538EB78E46506699FD9741637A50294494E1FFE9B6805C4F53
606D7576716658493A2E231C1715171C24303E506477889498917F654B3C4367A2DDF7DB
9046355E81684D8FD7A96D4C4A8BBF92436DB76677887B7F7E7E7E7E
7E7E7E7E7D808070908D576EAEA664377472E2D34C4275693C3F81D3FEF0BB7E52404352
626D706A60514234271D150F0C0A0A0D121A2534465A6F8290948D785C3F303C69AAE5F5
CB7C4248757A4D58B6CE8F64425EA8B869449E9A569576807E7E7E7E
7E7E7E7E7D83767B996E5895B38041587DA7F57B2B5D7956426FC1F8F5C17E4A3235465A
696F6B625446392F26211D1A18171716171A1F283445596E8290948B73533426386DB4EB
EEB66A476181643D79CEB58055437DBB9C4867B56879867C7F7E7E7E
7E7E7E7E8080728F895776B09B583B8585E2BB313978704E62ABEDF8C9824527283C5569
73726A5E52473F3B3938393A3B3A383632302E2F333C4A5D728695988B6F4A291E3775BF
EDE1A05F55797C4743A2CC9F7545539DBB7243999A599477807E7E7E
7E7E7E7D82777C976C5D9BAE723F6491AEE45D1B6183645D94DBF6D289441E1C33526C7A
7D766C61595453555A5F64686B6B69645D564E4743434854657A8E9C9D8D6C421F183A7F
C9EBCE895C6B88673262BFBA9362406FB6A44D63B3697B837D7E7E7E
7E7E7E7F80738E86597DB08F50439793DAA01A3A847E6481C5EFD9934718132C5071848A
857C726B686B7179838C949B9E9F9D978F84786B605652545D6E8398A5A590693A171541
8DD1E3B77865838648328BC4AA864D4A91BD7B43959A5B92777F7E7E
7E7E7E82787D946A62A0A7674171A3ADCF42156E937575ACE1DD9F4E170B244E768F9895
8B817B7A7E86929EABB6C0C7CBCCCAC5BDB1A4948374675F5E66778CA2B0AD9366321016
4C9BD5D49E6F7794722E4DABB9A1714062AFAB5360B1697D817E7E7E
7E7E7F80758D845B83AF844B4EA69ACB850B44988E7695CEDCAB5A19061D4C7B9AA7A49A
8F87868A94A1AFBDC9D4DDE3E7E8E7E3DDD3C7B8A694817268666D7E96ADBAB495612A0B
1B5BA9D4C089728E96512875B6B096574485BC834492995E90787F7E
7E7E81797D926967A39F5E467FAFA7B82C1783A68485B6D5B669210317487EA4B5B3A799
8F8C909AA6B4C2CED8DFE5E9ECEDEDECE8E3DACFC0AF9C89786C6971849EB8C5BB955A22
0A256CB4CCA87C81A283313C96B2AC804458A6B0585DAE6A7F807E7E
7F7F7F768C825D88AC7A495BB39CB96D0456AE9C819EC7BD7A2D0411437FADC1C0B2A093
8E9098A4B0BCC6CED3D6D8D9DADBDCDDDDDCDAD5CCC0B09D8A786C697389A6C2CEC09353
1B0C347FBABC917A98A7612461A5B0A4634179BA8B458F98618F797F
7E817A7E90696CA597574E8CB79BA11E239DB78B8AB2BD8B3D090C3D7EB3CDCCBBA5948B
8A919BA6B0B7BBBDBCBAB8B6B5B6B7BABEC2C6C8C7C2B9AC9A87756967738CAECBD7C38E
4915124690B9A78086AE983C3182A7B38E4B4F9DB35E5BAC6A817E7E
7A80778B805F8CA8714969BC99A45A066FC5A4829BB69951130A357AB6D6D7C3A8918481
868F98A0A4A5A39E98928C888585878C929AA3ABB2B5B5AFA494806E6262718FB5D5DDC2
863F131E5B9DB0907B9DB875285192ABAF70416EB791478C96638D7A
7F7B7E8E6970A58F525798BA8C8C1736B8C28B86A6A166220B2E73B6DDE0CAAA8E7C7679
81899092908A82796F655D575453555A626C7784919CA4A8A59B8C79665A5C6E91BCDDE2
BF7C36152E70A39F7E85B6AC4D2E6F99B49C544994B6635AAA6B827F
pop
P
% End of image
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[8]:=",
  ImageSize->{288, 288},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000182000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00;P3oool300000080oooo0P00000[0?ooo`@00000
0`3oool2000002d0oooo0P00000^0?ooo`@000000`3oool2000002d0oooo0`0000020?ooo`800000
7@3oool002l0oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`0X0?ooo`030000
003oool0oooo00<0oooo00@000000?ooo`3oool00000:`3oool010000000oooo0?ooo`00000]0?oo
o`030000003oool0oooo00<0oooo00@000000?ooo`3oool00000;@3oool010000000oooo0?ooo`00
00020?ooo`030000003oool0oooo01X0oooo000/0?ooo`D0000000D0oooo0000003oool0oooo0000
000[0?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool00000:`3oool010000000oooo
0?ooo`00000^0?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool00000:P3oool50000
00050?ooo`000000oooo0?ooo`000000703oool002H0oooo100000020?ooo`040000003oool0oooo
00000080oooo00@000000?ooo`3oool00000903oool4000000@0oooo00D000000?ooo`3oool0oooo
000000020?ooo`030000003oool0oooo02T0oooo00@000000?ooo`3oool00000;`3oool01@000000
oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0:03oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo000001`0oooo000]0?ooo`030000003oool000000080oooo00@00000
0?ooo`3oool00000:P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000002/0
oooo00@000000?ooo`3oool00000;@3oool010000000oooo0?ooo`0000020?ooo`040000003oool0
oooo000002/0oooo00<000000?ooo`0000000P3oool010000000oooo0?ooo`00000L0?ooo`00;P3o
ool2000000<0oooo0P00000/0?ooo`800000103oool2000002d0oooo0P00000_0?ooo`800000103o
ool2000002d0oooo0P0000030?ooo`8000007@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0
oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool000l0ooooo`00000A00000040oooo000?0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo0080oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`0L0?ooo`03
0000003oool0oooo0340oooo00<000000?ooo`3oool0<P3oool00`000000oooo0?ooo`0a0?ooo`03
0000003oool0oooo0340oooo00<000000?ooo`3oool0703oool100000040oooo000?0?ooo`030000
003oool0oooo0?l0oooo3@3oool100000040oooo00080?ooo`8000001@3oool00`000000oooo0?oo
o`3o0?ooo`d0oooo0@0000010?ooo`001`3oool010000000oooo0?ooo`0000040?ooo`030000003o
ool0oooo0?l0oooo3@3oool100000040oooo00070?ooo`040000003oool0oooo000000@0oooo0`00
003o0?ooo``0oooo0P0000010?ooo`001`3oool010000000oooo0?ooo`0000040?ooo`030000003o
ool0oooo0080ooooJ`1nOWh307moO`80SXj>0`3kno/80?ooo`80TY:B0`1oOgl807inOP80Ogmo0`2`
/;050?ooo`<0Xj>S0P1oOgm^07inOPD0oooo0@0000010?ooo`001`3oool010000000oooo0?ooo`00
00040?ooo`030000003oool0oooo0080ooooJ`1nOWh307moO`80SXj>0`3kno/80?ooo`80TY:B0`1o
Ogl807inOP80Ogmo0`2`/;050?ooo`<0Xj>S0P1oOgm^07inOPD0oooo0@0000010?ooo`00203oool2
000000D0oooo00<000000?ooo`3oool00P3ooom[07inOP<0Ogmo0P2>SXh30?_kn`P0oooo0P2BTY83
07moO`P0OWin0P1oOgl30;2`/0D0oooo0`2SXj<207moOfh0OWin1@3oool100000040oooo000?0?oo
o`030000003oool0oooo0080ooooJ`1nOWh30820P080UIFE2`3oool209^KV`<0P820201nOWh208:2
PP<0bl_;1@3oool30;bl_080PH61KP1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3o
ool00P3ooom[07inOP<0P8200P2EUID;0?ooo`80Vi^K0`20P80807inOP80PX:20`3;bl/50?ooo`<0
_;bl0P21PH5^07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooof/0OWin
0`23Ph<20:ZZZP<0o?cl203oool20;6a/@<0QHF50P1oOgl607inOP80Shn?0`3jn_X30=gMg@80jNWY
0`3^k^h208Z:RVh0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooJ`1n
OWh308>3P`80ZZZZ0`3lo?`80?ooo`80/K6a0`25QHD207moO`H0OWin0P2?Shl30?[jnP<0gMgM0P3Y
jNT30>k^kP80RXZ:KP1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooom[
07inOP<0Ph>30P2ZZZX30?clo0P0oooo0P2a/K4308F5Q@80Ogmo1P1nOWh208n?S`<0n_[j0`3MgMd2
0>WYj@<0k^k^0P2:RXY^07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oo
ofP0OWin0`1oOgl308b<S080`L710`3?cll205IFEP<0Bd];0`38b<P20<O7a`<0T92@0P20P80307in
OP<0P8200P2a/K430>c/k0<09RHV0P0j>SX30?;blP80YjNW0`20P81[07inOPD0oooo0@0000010?oo
o`003`3oool00`000000oooo0?ooo`020?ooofP0OWin0`1oOgl308b<S080`L710`3?cll205IFEP<0
Bd];0`38b<P20<O7a`<0T92@0P20P80307inOP<0P8200P2a/K430>c/k0<09RHV0P0j>SX30?;blP80
YjNW0`20P81[07inOPD0oooo0@0000010?ooo`003`3oool2000000<0ooooJ01nOWh308:2PP<0W9bL
0P34a<@306i^KPD000000`1MGEd20<C4a0<0XJ6Q0P23Ph<307moO`<0RXZ:0P3Fe]H307alO0D00000
0`2CTi<20<k>cP<0QXJ6J`1nOWh40?ooo`8000000@3oool000l0oooo00<000000?ooo`3oool00P3o
oomX07inOP<0PX:20`2LW9`20<C4a0<0KVi^1@00000305eMG@80a<C40`2QXJ4208>3P`<0Ogmo0`2:
RXX20=KFeP<0O7al1@00000309>CT`80c/k>0`26QXI[07inOPD0oooo0@0000010?ooo`003`3oool0
0`000000oooo0?ooo`020?ooofP0OWin0`22PX8309bLW080a<C40`1^KVh5000000<0GEeM0P34a<@3
0:6QX@80Ph>30`1oOgl308Z:RP80e]KF0`1lO7`5000000<0Ti>C0P3>c/h308J6QV/0OWin1@3oool1
00000040oooo000?0?ooo`030000003oool0oooo0080ooooIP1nOWh207moO`<0R8R80`2[Zj/209fM
W@/000000P2BTY830:j^[P80RXZ:0`21PH430::RXP80a<C42000000300P82080c<c<0`2KVi/307mo
OfP0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooIP1nOWh207moO`<0
R8R80`2[Zj/209fMW@/000000P2BTY830:j^[P80RXZ:0`21PH430::RXP80a<C42000000300P82080
c<c<0`2KVi/307moOfP0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo
IP1nOWh207moO`<0R8R80`2[Zj/209fMW@/000000P2BTY830:j^[P80RXZ:0`21PH430::RXP80a<C4
2000000300P82080c<c<0`2KVi/307moOfP0OWin1@3oool100000040oooo000?0?ooo`030000003o
ool0oooo0080ooooIP1nOWh20861P@<0Ti>C0`2XZ:P204Y:BP/000000P0i>CT30:JVYP80UYJF0`29
RHT30;fm_@80G5aL2`00000207=cL`<0^;Rh0`24Q8AX07inOPD0oooo0@0000010?ooo`003`3oool0
0`000000oooo0?ooo`020?ooofH0OWin0P21PH4309>CT`<0Z:RX0P1:BTX;00000080>CTi0`2VYZH2
09JFUP<0RHV90`2m_Kd205aLG0/000000P1cLg<30;Rh^0<0Q8B4J01nOWh50?ooo`4000000@3oool0
00l0oooo00<000000?ooo`3oool00P3ooomS07inOP<0Ogmo0P25QHD309bLW0<0Ph>33`00000307Ui
N@80WinO0`2PX:030:^[Z`D000000`030`<7000000<0/k>c0`2BTY8207moOfH0OWin1@3oool10000
0040oooo000?0?ooo`030000003oool0oooo0080ooooH`1nOWh307moO`80QHF50`2LW9`308>3P`l0
00000`1iNGT209nOW`<0X:2P0`2[Zj/5000000<00`<31`0000030;>c/`<0TY:B0P1oOgmV07inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooof<0OWin0`1oOgl208F5Q@<0W9bL
0`23Ph<?000000<0NGUi0P2OWil30:2PX0<0Zj^[1@00000300<30`L000000`2c/k<309:BTP80Ogmo
IP1nOWh50?ooo`4000000@3oool000l0oooo0P0000030?ooof<0OWin0`20P80208b<S0<0UiNG0`0n
?Sh5000000806QXJ0`0M7Ad5000000<0<30`0P2IVIT30;bl_0<0DE5A0P00000301DE5@<0C4a<0P1:
BTX300`<308000000`1SHf<30:RXZ080PX:2IP1nOWh40?ooo`8000000@3oool000l0oooo00<00000
0?ooo`3oool00P3ooomS07inOP<0P8200P2<S8`309NGU`<0?Shn1@00000201XJ6P<07AdM1@000003
030`<080VIVI0`2l_;`3055AD@8000000`0E5AD304a<C080BTY:0`0<30`2000000<0Hf=S0`2XZ:P2
08:2PVH0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooH@1nOWh207mo
O`<0Ph>30P2CTi<307IfMPD000000`0o?cl208F5Q@<0R8R80`19BDT500000080OWin0`2`/;030000
00800@410`1[Jf/309FEU@80Ti>C0`1SHf<5000000<0XZ:R0P2<S8`307moOf<0OWin1@3oool10000
0040oooo000?0?ooo`030000003oool0oooo0080ooooH@1nOWh207moO`<0Ph>30P2CTi<307IfMPD0
00000`0o?cl208F5Q@<0R8R80`19BDT500000080OWin0`2`/;03000000800@410`1[Jf/309FEU@80
Ti>C0`1SHf<5000000<0XZ:R0P2<S8`307moOf<0OWin1@3oool100000040oooo000?0?ooo`030000
003oool0oooo0080ooooH@1nOWh207moO`<0Ph>30P2CTi<307IfMPD000000`0o?cl208F5Q@<0R8R8
0`19BDT500000080OWin0`2`/;03000000800@410`1[Jf/309FEU@80Ti>C0`1SHf<5000000<0XZ:R
0P2<S8`307moOf<0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooH@1n
OWh207moO`<0R8R80P2<S8`303hn?P<000000P0i>CT30:^[Z`80hN7Q0`3Shn<30;Bd]080ATI60`00
000205ADE0<0EUIF0`00000206=SH`<0][Jf0`3@d=020<o?c`<0/K6a0P1DE5@3000000<0GUiN0P2M
WId30820P6<0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooH@1nOWh2
07moO`<0R8R80P2<S8`303hn?P<000000P0i>CT30:^[Z`80hN7Q0`3Shn<30;Bd]080ATI60`000002
05ADE0<0EUIF0`00000206=SH`<0][Jf0`3@d=020<o?c`<0/K6a0P1DE5@3000000<0GUiN0P2MWId3
0820P6<0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooH@1nOWh207mo
O`<0R8R80P2<S8`303hn?P<000000P0i>CT30:^[Z`80hN7Q0`3Shn<30;Bd]080ATI60`00000205AD
E0<0EUIF0`00000206=SH`<0][Jf0`3@d=020<o?c`<0/K6a0P1DE5@3000000<0GUiN0P2MWId30820
P6<0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooH@1nOWh20861P@<0
S8b<0P1`L703014A4@<08B4Q0P2VYZH;0?ooo`80/k>c0`0i>CT201hN7P<000000`13@d<20;fm_@<0
j^[Z1@3emOD30>OWi`80/;2`0`0T92@300L71`80UiNG0`28R8QS07inOPD0oooo0@0000010?ooo`00
3`3oool00`000000oooo0?ooo`020?ooof40OWin0P21PH4308b<S080L71`0`0A4A43024Q8@80YZJV
2`3oool20;>c/`<0>CTi0P0N7Qh3000000<0@d=30P2m_Kd30>[ZjPD0mOGe0`3WinL20;2`/0<092@T
0`071`L209NGU`<0R8R8H`1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3o
oomN07inOP<0Ogmo0P25QHD308J6QP80A4A40`0E5AD308>3P`l0oooo0`2ZZZX5000000<0Xj>S0P3j
n_X;0?ooo`80k^k^0`2?Shl300000080GEeM0`2DU9@20820P640OWin1@3oool100000040oooo000?
0?ooo`030000003oool0oooo0080ooooGP1nOWh307moO`80QHF50`26QXH204A4A0<05ADE0`23Ph<?
0?ooo`<0ZZZZ1@0000030:>SX`80n_[j2`3oool20>k^kP<0Shn?0`00000205eMG@<0U9BD0P20P81Q
07inOPD0oooo0@0000010?ooo`003`3oool2000000<0ooooGP1nOWh307moO`80QHF50`26QXH204A4
A0<05ADE0`23Ph<?0?ooo`<0ZZZZ1@0000030:>SX`80n_[j2`3oool20>k^kP<0Shn?0`00000205eM
G@<0U9BD0P20P81Q07inOP@0oooo0P0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooeh0
OWin0`20P80208R8R0<0KVi^0P0V9RH305EEE@<0i>CT3`3oool30?gmo@8000000`030`<30=CDe080
oooo0`3lo?`30>o_k`80l?3`0`3kno/20?ooo`<0hn?S0`1=CDd201@D50<0T92@0P25QHEQ07inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooeh0OWin0`20P80208R8R0<0KVi^
0P0V9RH305EEE@<0i>CT3`3oool30?gmo@8000000`030`<30=CDe080oooo0`3lo?`30>o_k`80l?3`
0`3kno/20?ooo`<0hn?S0`1=CDd201@D50<0T92@0P25QHEQ07inOPD0oooo0@0000010?ooo`003`3o
ool00`000000oooo0?ooo`020?oooeh0OWin0`20P80208R8R0<0KVi^0P0V9RH305EEE@<0i>CT3`3o
ool30?gmo@8000000`030`<30=CDe080oooo0`3lo?`30>o_k`80l?3`0`3kno/20?ooo`<0hn?S0`1=
CDd201@D50<0T92@0P25QHEQ07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`02
0?oooe/0OWin0`1oOgl308>3P`80PX:20`1<C4`203Lg=`<0/k>c203oool20?KfmP<0mOGe1@3oool3
0?Ogm`80H61P0`27QhL30<k>cP80moOg0`3Ogml30<[:bP80bl_;0`3Lg=`20?Shn0<0oooo0`2j^[X2
01@D50<0H61P0P2>SXh307moOeh0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo
0080ooooF`1nOWh307moO`<0Ph>30P22PX8304a<C080=cLg0`2c/k<80?ooo`80m_Kf0`3emOD50?oo
o`<0moOg0P1PH60308N7Q`<0c/k>0P3gmoL30=oOg`<0b/[:0P3;bl/30=cLg080n?Sh0`3oool30;Zj
^P8051@D0`1PH60208j>SP<0OgmoGP1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3o
ool00P3ooomK07inOP<0P8200`25QHD206m_K`<0>CTi0P1jNWX60?ooo`80nOWi0`3;bl/20;:b/P<0
/K6a0`3>c/h20?knoP<0^k^k1@3oool30;6a/@80]kNg0`2`/;0309ZJVP80VIVI0`2/[:`20=3@d0<0
o?cl0`3no_h206e]K@<08b<S0P2<S8`308>3Peh0OWin1@3oool100000040oooo000?0?ooo`030000
003oool0oooo0080ooooF`1nOWh30820P0<0QHF50P1_Kfl303Ti>@80NWYj1P3oool20?Win@<0bl_;
0P2b/[830;6a/@<0c/k>0P3no_h30;^k^`D0oooo0`2a/K420;Ng]`<0/;2`0`2JVYX209VIV@<0[:b/
0P3@d=030?clo0<0o_kn0P1]KFd302<S8`80S8b<0`23Ph=N07inOPD0oooo0@0000010?ooo`003`3o
ool00`000000oooo0?ooo`020?oooe/0OWin0`20P80308F5Q@80Kfm_0`0i>CT207YjNPH0oooo0P3i
nOT30<_;b`80/[:b0`2a/K430<k>cP80o_kn0`2k^k/50?ooo`<0/K6a0P2g]kL30;2`/0<0VYZJ0P2I
VIT30:b/[080d=3@0`3lo?`30?knoP80KFe]0`0S8b<208b<S0<0Ph>3GP1nOWh50?ooo`4000000@3o
ool000l0oooo00<000000?ooo`3oool00P3ooomI07inOP80Ogmo0`22PX830820P080E5AD0`1ADE42
0<o?c`<0oooo0`3dm?@20;6a/@<0P8201@1WIfL308Z:RP80QXJ60`2RXZ850?ooo`<0Y:BT0P1UIFD3
07=cL`<0J6QX0P1VIVH307IfMP80VIVI0`3@d=030?ooo`80eMGE0`0Z:RX206ATI0<0RXZ:0`1oOgmK
07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooeT0OWin0P1oOgl308:2
PP<0P8200P1DE5@3055AD@80clo?0`3oool30?Cdm080/K6a0`20P80506MWI`<0RXZ:0P26QXH30::R
XPD0oooo0`2TY:@206EUI@<0Lg=c0`1XJ6P206IVIP<0MWIf0P2IVIT30=3@d0<0oooo0P3EeMD302XZ
:P80I6AT0`2:RXX307moOe/0OWin1@3oool100000040oooo000?0?ooo`8000000`3ooomI07inOP80
Ogmo0`23Ph<3071`L080BTY:0`2CTi<20?ooo`<0no_k0`2ZZZX206=SH`<0>CTi0P0W9bL302XZ:P<0
?Shn0P0820P30;Zj^PD0oooo0`2^[Zh202PX:0<0<C4a0`0n?Sh203/k>`<0ADE50P1PH60309BDU0<0
g]kN0P3oool308B4Q080<30`0`29RHT30861PE/0OWin103oool200000040oooo000?0?ooo`030000
003oool0oooo0080ooooF@1nOWh207moO`<0Ph>30`1`L70204Y:BP<0Ti>C0P3oool30?_kn`<0ZZZZ
0P1SHf<303Ti>@809bLW0`0Z:RX303hn?P8020P80`2j^[X50?ooo`<0[Zj^0P0X:2P3034a<@<0?Shn
0P0k>c/304E5A@80H61P0`2DU9@30=kNgP80oooo0`24Q8@2030`<0<0RHV90`21PH5K07inOPD0oooo
0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooeT0OWin0P1oOgl308>3P`<0L71`0P1:
BTX309>CT`80oooo0`3kno/30:ZZZP80Hf=S0`0i>CT202LW9`<0:RXZ0`0n?Sh200P820<0^[Zj1@3o
ool30:j^[P80:2PX0`0a<C4303hn?P80>c/k0`15ADD2061PH0<0U9BD0`3Ng]h20?ooo`<0Q8B40P0`
<30308V9R@<0PH61F`1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooomI
07inOP80PH610`1oOgl305aLG080IFEU0`3Jf]X20?ooo`<0/k>c0`1IFET2020P80<01@D50P000003
00D51@D000000`33`l<50?ooo`<0^KVi0P0O7al3000000<07QhN0P0R8R8302@T9080<c<c0`1HF5P3
09fMW@80m?Cd0`3Ti>@203`l?0<0J6QX0`27QhL207moOeT0OWin1@3oool100000040oooo000?0?oo
o`030000003oool0oooo0080ooooF@1nOWh20861P@<0Ogmo0`1LG5`206EUI@<0f][J0P3oool30;>c
/`<0FEUI0P0P820300D51@8000000`051@D5000000<0`l?31@3oool30;Vi^@807alO0`00000301hN
7P808R8R0`0T92@203<c<`<0F5QH0`2MWId20?Cdm0<0i>CT0P0l?3`306QXJ0<0QhN70P1oOgmI07in
OPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooeT0OWin0P21PH4307moO`<0
G5aL0P1UIFD30=[JfP80oooo0`2c/k<305UIF@80820P0`051@D2000000<01@D51@0000030<?3``D0
oooo0`2i^KT201lO7`<000000`0N7Qh2028R8P<092@T0P0c<c<305QHF0<0WIfM0P3dm?@30>CTi080
?3`l0`1XJ6P308N7Q`80OgmoF@1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool0
0P3ooomF07inOP<0Ogmo0P21PH43079bLP<0EeMG0P2QXJ430?_kn`80bLW90`1PH60301LG5`d00000
0P0M7Ad309NGU`80RHV90`2KVi/30:RXZ080AdM70`00000300X:2P8071`L0`0I6AT201XJ6P<0:b/[
0`1KFe/20;Jf]P<0oooo0P2DU9@303dm?@<0QXJ60P20P81I07inOPD0oooo0@0000010?ooo`003`3o
ool00`000000oooo0?ooo`020?oooeH0OWin0`1oOgl20861P@<0LW9b0`1GEeL20:6QX@<0no_k0P39
bLT3061PH0<05aLG3@00000201dM7@<0UiNG0P29RHT309^KV`<0Z:RX0P17AdL3000000<02PX:0P0L
71`301TI6@806QXJ0`0[:b/305]KF`80][Jf0`3oool209BDU0<0?Cdm0`26QXH20820P5T0OWin1@3o
ool100000040oooo000?0?ooo`030000003oool0oooo0080ooooEP1nOWh30820P080OWin0`1SHf<3
07=cL`80fm_K0`3QhN4207IfMP<06QXJ2000000200D51@H000000P1TI6@304M7A`8000000`0H61P3
07=cL`80OWin0`0M7Ad300T92@809RHV0`0S8b<201TI6@<05QHF0`0Z:RX206a/K0<0f][J0P3Zj^X3
04Y:BP<0K6a/0P25QHEI07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oo
oeH0OWin0`20P80207inOP<0Hf=S0`1cLg<20=_Kf`<0hN7Q0P1fMWH301XJ6PP000000P051@D60000
0080I6AT0`17AdL2000000<061PH0`1cLg<207inOP<07AdM0`092@T202HV9P<08b<S0P0I6AT301HF
5P<0:RXZ0P1/K6`30=[JfP80j^[Z0`1:BTX306a/K080QHF5F@1nOWh50?ooo`4000000@3oool00003
0?ooo`000000000000<000000P3oool2000000D0oooo00<000000?ooo`3oool00P3ooomF07inOP<0
P8200P1nOWh306=SH`<0Lg=c0P3Kfm/30>7Qh@80MWIf0`0J6QX8000000801@D51P00000206ATI0<0
AdM70P00000301PH60<0Lg=c0P1nOWh301dM7@<02@T90P0V9RH302<S8`806ATI0`0F5QH302XZ:P80
K6a/0`3Jf]X20>[ZjP<0BTY:0`1/K6`208F5QET0OWin1@3oool100000040oooo00030?ooo`050000
003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080
ooooE01nOWh207moO`<0P8200P1dM7@3065QH@<0Z:RX0P3YjNT309VIV@80:RXZ1P00000200H61P<0
71`L0P0[:b/3000000<0Bd];0P1nOWh8000000<0;bl_0P2IVIT307=cL`<0:b/[0P0j>SX30410@080
;Rh^0`0K6a/301@D5080<C4a0`2=SHd20?Ogm`<0WYjN0`18B4P208F5Q@<0P820EP1nOWh50?ooo`40
00000@3oool000<0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo0080
oooo0`0000020?oooe@0OWin0P1oOgl30820P080M7Ad0`1QHF430:RXZ080jNWY0`2IVIT202XZ:PH0
00000P061PH301`L7080:b/[0`00000304];B`80OWin2000000302l_;`80VIVI0`1cLg<302/[:`80
>SXj0`10@40202h^;P<06a/K0`0D51@2034a<@<0SHf=0P3gmoL309jNWP<0B4Q80P25QHD30820P5H0
OWin103oool200000040oooo00030?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo0080ooooE01nOWh207moO`<0OWin0P1XJ6P307alO0<0
eMGE0P2n_[h304M7A`D000000`0D51@203Lg=`<0EeMG0P1;Bd/303dm?@<0aLG50P1_Kfl;00000080
SXj>0`34a<@307EeM@80F5QH0`1UIFD205ADE0<0=cLg0`0K6a/201<C4`<0A4A40P2m_Kd30>[ZjP<0
EEEE0P1`L70308>3PeH0OWin1@3oool100000040oooo00030?ooo`050000003oool0oooo0?ooo`00
00000P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080ooooE01nOWh207moO`<0
OWin0P1XJ6P307alO0<0eMGE0P2n_[h304M7A`D000000`0D51@203Lg=`<0EeMG0P1;Bd/303dm?@<0
aLG50P1_Kfl;00000080SXj>0`34a<@307EeM@80F5QH0`1UIFD205ADE0<0=cLg0`0K6a/201<C4`<0
A4A40P2m_Kd30>[ZjP<0EEEE0P1`L70308>3PeH0OWin1@3oool100000040oooo00020?ooo`800000
103oool2000000D0oooo00<000000?ooo`3oool00P3ooomD07inOP80Ogmo0`1nOWh206QXJ0<0O7al
0`3EeMD20;jn_P<0AdM71@00000301@D5080=cLg0`1GEeL204];B`<0?Cdm0`35aLD206m_K`/00000
0P2>SXh30<C4a0<0MGEe0P1HF5P306EUI@80E5AD0`0g=cL301/K6`804a<C0`14A4@20;fm_@<0j^[Z
0`1EEED2071`L0<0Ph>3EP1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3o
oomA07inOP<0Ogmo0P20P80307IfMP80JFUY0`2ZZZX30=CDe080LG5a0`0<30`2000000<071`L0`1<
C4`207IfMP<0U9BD0P1ZJVX30=3@d0<0m?Cd0P1IFET;00000080Jf][0`3Vi^H30=3@d080QhN70`28
R8P20861P@<0HV9R0`0k>c/201LG5`<061PH0P1YJFT30>WYj@<0XZ:R0P1ADE4308>3P`80OgmoE01n
OWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooomA07inOP<0Ogmo0P20P803
07IfMP80JFUY0`2ZZZX30=CDe080LG5a0`0<30`2000000<071`L0`1<C4`207IfMP<0U9BD0P1ZJVX3
0=3@d0<0m?Cd0P1IFET;00000080Jf][0`3Vi^H30=3@d080QhN70`28R8P20861P@<0HV9R0`0k>c/2
01LG5`<061PH0P1YJFT30>WYj@<0XZ:R0P1ADE4308>3P`80OgmoE01nOWh50?ooo`4000000@3oool0
00l0oooo00<000000?ooo`3oool00P3ooomA07inOP<0Ogmo0P20P80307IfMP80JFUY0`2ZZZX30=CD
e080LG5a0`0<30`2000000<071`L0`1<C4`207IfMP<0U9BD0P1ZJVX30=3@d0<0m?Cd0P1IFET;0000
0080Jf][0`3Vi^H30=3@d080QhN70`28R8P20861P@<0HV9R0`0k>c/201LG5`<061PH0P1YJFT30>WY
j@<0XZ:R0P1ADE4308>3P`80OgmoE01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3o
ool00P3ooomA07inOP<0Ogmo0P1nOWh306e]K@80PX:20`3;bl/30:2PX080:2PX0`00000201/K6`<0
EeMG0`2<S8`20;Fe]@<0]kNg0P2f][H30?ooo`<0h>3P0P1HF5P3028R8P804a<C0`0820P300d=3@80
CDe=0`3@d=030?ooo`80alO70`2TY:@20:RXZ0<0TI6A0`1YJFT203Ph>0<04Q8B0P0X:2P30::RXP<0
i^KV0P1MGEd307AdM080PX:2E01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool0
0P3ooomA07inOP<0Ogmo0P1nOWh306e]K@80PX:20`3;bl/30:2PX080:2PX0`00000201/K6`<0EeMG
0`2<S8`20;Fe]@<0]kNg0P2f][H30?ooo`<0h>3P0P1HF5P3028R8P804a<C0`0820P300d=3@80CDe=
0`3@d=030?ooo`80alO70`2TY:@20:RXZ0<0TI6A0`1YJFT203Ph>0<04Q8B0P0X:2P30::RXP<0i^KV
0P1MGEd307AdM080PX:2E01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3o
oomA07inOP<0Ogmo0P1hN7P306m_K`80ZJVY0`2o_kl305=CD`800`<30`0E5AD205QHF0<0VIVI0`37
alL20>CTi0<0`l?31@3oool30;>c/`80LG5a0`1iNGT208F5Q@<0MGEe0`1ADE4204Y:BP<0Vi^K1@3o
ool30<71`@80`<300`2i^KT309ZJVP80JFUY0`0_;bl200l?3`<0CDe=0`3IfMT20:BTY0<0FEUI0P22
PX8307moOe40OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooD@1nOWh3
07moO`80N7Qh0`1_Kfl20:VYZ@<0_kno0`1CDe<200<30`<05ADE0P1HF5P309VIV@<0alO70P3Ti>@3
0<?3``D0oooo0`2c/k<2075aL@<0NGUi0P25QHD307EeM@<0DE5A0P1:BTX309^KV`D0oooo0`31`L42
0<30`0<0^KVi0`2JVYX206UYJ@<0;bl_0P0?3`l304e=C@<0fMWI0P2TY:@305UIF@80PX:20`1oOgmA
07inOPD0oooo0@0000010?ooo`003`3oool2000000<0ooooD@1nOWh307moO`80N7Qh0`1_Kfl20:VY
Z@<0_kno0`1CDe<200<30`<05ADE0P1HF5P309VIV@<0alO70P3Ti>@30<?3``D0oooo0`2c/k<2075a
L@<0NGUi0P25QHD307EeM@<0DE5A0P1:BTX309^KV`D0oooo0`31`L420<30`0<0^KVi0`2JVYX206UY
J@<0;bl_0P0?3`l304e=C@<0fMWI0P2TY:@305UIF@80PX:20`1oOgmA07inOP@0oooo0P0000010?oo
o`003`3oool00`000000oooo0?ooo`020?ooodh0OWin0`1oOgl307inOP80LG5a0`25QHD20<30`0<0
QXJ60`0H61P200d=3@<0Cdm?0P2LW9`30=7Ad@<0l?3`0P3/k>`30>GUi@80oooo0`3kno/3096AT@80
W9bL0`3AdM420?3`l0<0gmoO0`2TY:@206EUI@<0JVYZ0`3Cdm<20?ooo`<0hN7Q0P37alL30=3@d0<0
`L710P2LW9`3065QH@808B4Q0`0G5aL308Z:RP80gmoO0`1SHf<207IfMP<0PH61D@1nOWh50?ooo`40
00000@3oool000l0oooo00<000000?ooo`3oool00P3ooom>07inOP<0Ogmo0`1nOWh2075aL@<0QHF5
0P30`<0308J6QP<061PH0P0=3@d304m?C`80W9bL0`3AdM430?3`l080k>c/0`3UiND20?ooo`<0no_k
0`2ATI4209bLW0<0dM7A0P3`l?030=oOg`<0Y:BT0P1UIFD306YZJP<0dm?C0P3oool30>7Qh@80alO7
0`3@d=030<71`@80W9bL0`1QHF42024Q8@<05aLG0`2:RXX20=oOg`<0Hf=S0P1fMWH30861PE40OWin
1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooCP1nOWh307moO`<0NGUi0P1c
Lg<30:JVYP80[Jf]0`10@40300T92@80?clo0`2DU9@20=CDe0<0lo?c0`3moOd20=;BdPD0oooo0`2L
W9`308B4Q080bLW9203oool30>k^kP80Ti>C0`1DE5@307inOP80kno_0`3gmoL20<K6aP<0d=3@0`3G
emL20<G5a@<0V9RH0P1ADE4301<C4`<0=cLg0P39bLT30:>SX`80H61P0`22PX8307moOdh0OWin1@3o
ool100000040oooo000?0?ooo`030000003oool0oooo0080ooooCP1nOWh307moO`<0NGUi0P1cLg<3
0:JVYP80[Jf]0`10@40300T92@80?clo0`2DU9@20=CDe0<0lo?c0`3moOd20=;BdPD0oooo0`2LW9`3
08B4Q080bLW9203oool30>k^kP80Ti>C0`1DE5@307inOP80kno_0`3gmoL20<K6aP<0d=3@0`3GemL2
0<G5a@<0V9RH0P1ADE4301<C4`<0=cLg0P39bLT30:>SX`80H61P0`22PX8307moOdh0OWin1@3oool1
00000040oooo000?0?ooo`030000003oool0oooo0080ooooCP1nOWh307moO`<0NGUi0P1cLg<30:JV
YP80[Jf]0`10@40300T92@80?clo0`2DU9@20=CDe0<0lo?c0`3moOd20=;BdPD0oooo0`2LW9`308B4
Q080bLW9203oool30>k^kP80Ti>C0`1DE5@307inOP80kno_0`3gmoL20<K6aP<0d=3@0`3GemL20<G5
a@<0V9RH0P1ADE4301<C4`<0=cLg0P39bLT30:>SX`80H61P0`22PX8307moOdh0OWin1@3oool10000
0040oooo000?0?ooo`030000003oool0oooo0080ooooC01nOWh207moO`<0OWin0`1dM7@208N7Q`<0
][Jf0P1cLg<301DE5@<0:RXZ0P24Q8@30=3@d080m?Cd0`3inOT30>7Qh@80alO70`3oool20;6a/@<0
EUIF0`29RHT20>OWi`/0oooo0P32`/8305iNGP<0>c/k0P2DU9@30>OWi`80a/K60`2l_;`30=CDe080
fMWI0`33`l<208^;R`<0>SXj0`0A4A4207IfMP<0emOG0P1XJ6P307UiN@<0P820CP1nOWh50?ooo`40
00000@3oool000l0oooo00<000000?ooo`3oool00P3ooom<07inOP80Ogmo0`1nOWh307AdM080QhN7
0`2f][H207=cL`<05ADE0`0Z:RX208B4Q0<0d=3@0P3dm?@30?Win@<0hN7Q0P37alL30?ooo`80/K6a
0`1FEUH308V9R@80inOW2`3oool20<;2`P<0GUiN0`0k>c/209BDU0<0inOW0P36a/H30;bl_0<0e=CD
0P3IfMT30<?3``80Rh^;0`0j>SX3014A4@80MWIf0`3GemL206QXJ0<0NGUi0`20P81>07inOPD0oooo
0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oood`0OWin0P1oOgl307inOP<0M7Ad0P27
QhL30;Jf]P80Lg=c0`0E5AD302XZ:P80Q8B40`3@d=020?Cdm0<0nOWi0`3QhN420<O7a`<0oooo0P2a
/K4305IFEP<0RHV90P3WinL;0?ooo`80`/;20`1NGUh303/k>`80U9BD0`3WinL20<K6aP<0_;bl0`3D
e=@20=WIf@<0`l?30P2;Rh/303Xj>P<04A4A0P1fMWH30=OGe`80J6QX0`1iNGT30820P4h0OWin1@3o
ool100000040oooo000?0?ooo`030000003oool0oooo0080ooooC01nOWh207moO`<0NWYj0`1fMWH2
0::RXP<0WYjN0P0e=CD301/K6`<0Jf][0P34a<@30?;blP80mOGe0`3Rh^830:NWY`80f=SH0`3?cll5
03Lg=`<0Ti>C0P3Zj^X;0?ooo`80gmoO0`1lO7`3028R8P80<c<c0`2TY:@20;no_`<0Xj>S0`2k^k/2
0=GEe@<0fMWI0P2m_Kd307IfMP<08B4Q0P0Y:BT30;^k^`80XJ6Q0`1VIVH30861PDh0OWin1@3oool1
00000040oooo000?0?ooo`030000003oool0oooo0080ooooC01nOWh207moO`<0NWYj0`1fMWH20::R
XP<0WYjN0P0e=CD301/K6`<0Jf][0P34a<@30?;blP80mOGe0`3Rh^830:NWY`80f=SH0`3?cll503Lg
=`<0Ti>C0P3Zj^X;0?ooo`80gmoO0`1lO7`3028R8P80<c<c0`2TY:@20;no_`<0Xj>S0`2k^k/20=GE
e@<0fMWI0P2m_Kd307IfMP<08B4Q0P0Y:BT30;^k^`80XJ6Q0`1VIVH30861PDh0OWin1@3oool10000
0040oooo000?0?ooo`8000000`3ooom>07inOP<0MWIf0`27QhL20:b/[0<0IFEU0P0J6QX304e=C@<0
/;2`0P3/k>`30??cl`80fMWI0`2[Zj/30820P080b<S80`12@T82000000<0>CTi0`2GUiL20<o?c`<0
j>SX0P3bl_860?[jnP80hN7Q0`2KVi/303@d=08000000`13@d<209jNWP<0T92@0`2DU9@20;^k^`<0
eMGE0P3Hf=P30:n_[`<0EeMG0P0B4Q8306MWI`80c/k>0`1[Jf/307YjNP80OgmoC01nOWh40?ooo`80
00000@3oool000l0oooo00<000000?ooo`3oool00P3ooom>07inOP<0MWIf0`27QhL20:b/[0<0IFEU
0P0J6QX304e=C@<0/;2`0P3/k>`30??cl`80fMWI0`2[Zj/30820P080b<S80`12@T82000000<0>CTi
0`2GUiL20<o?c`<0j>SX0P3bl_860?[jnP80hN7Q0`2KVi/303@d=08000000`13@d<209jNWP<0T92@
0`2DU9@20;^k^`<0eMGE0P3Hf=P30:n_[`<0EeMG0P0B4Q8306MWI`80c/k>0`1[Jf/307YjNP80Ogmo
C01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooom>07inOP<0MWIf0`27
QhL20:b/[0<0IFEU0P0J6QX304e=C@<0/;2`0P3/k>`30??cl`80fMWI0`2[Zj/30820P080b<S80`12
@T82000000<0>CTi0`2GUiL20<o?c`<0j>SX0P3bl_860?[jnP80hN7Q0`2KVi/303@d=08000000`13
@d<209jNWP<0T92@0`2DU9@20;^k^`<0eMGE0P3Hf=P30:n_[`<0EeMG0P0B4Q8306MWI`80c/k>0`1[
Jf/307YjNP80OgmoC01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooom9
07inOP<0Ogmo0P1kNg/307UiN@<0WYjN0P2ATI45034a<@<0Ti>C0`3Rh^820?;blP<0eMGE0P2XZ:P3
06=SH`<0PX:20P1[Jf/5000000<0CDe=0`2=SHd209fMW@<0V9RH0P2EUID30:2PX0<0][Jf0P34a<@3
0:f][@<0GEeM1@00000205aLG0<0P8200`1`L702096AT@<0^k^k0P3Hf=P30=CDe0<0V9RH0P0d=3@3
028R8P80[jn_0`2MWId306][J`80P820C01nOWh50?ooo`4000000@3oool000l0oooo00<000000?oo
o`3oool00P3ooom907inOP<0Ogmo0P1kNg/307UiN@<0WYjN0P2ATI45034a<@<0Ti>C0`3Rh^820?;b
lP<0eMGE0P2XZ:P306=SH`<0PX:20P1[Jf/5000000<0CDe=0`2=SHd209fMW@<0V9RH0P2EUID30:2P
X0<0][Jf0P34a<@30:f][@<0GEeM1@00000205aLG0<0P8200`1`L702096AT@<0^k^k0P3Hf=P30=CD
e0<0V9RH0P0d=3@3028R8P80[jn_0`2MWId306][J`80P820C01nOWh50?ooo`4000000@3oool000l0
oooo00<000000?ooo`3oool00P3ooom907inOP<0Ogmo0P1kNg/307UiN@<0WYjN0P2ATI45034a<@<0
Ti>C0`3Rh^820?;blP<0eMGE0P2XZ:P306=SH`<0PX:20P1[Jf/5000000<0CDe=0`2=SHd209fMW@<0
V9RH0P2EUID30:2PX0<0][Jf0P34a<@30:f][@<0GEeM1@00000205aLG0<0P8200`1`L702096AT@<0
^k^k0P3Hf=P30=CDe0<0V9RH0P0d=3@3028R8P80[jn_0`2MWId306][J`80P820C01nOWh50?ooo`40
00000@3oool000l0oooo00<000000?ooo`3oool00P3ooom<07inOP80N7Qh0`27QhL30:>SX`80GEeM
0`0U9BD206i^KP<0clo?0`3alO420=KFeP<0XZ:R0P1XJ6P303Ti>@<0N7Qh1@000002014A4@<0HV9R
0`1fMWH2061PH0<0A4A40P0f=SH30492@P<0IFEU0P2ATI430:RXZ0<0R8R80P0Y:BT3000000803Ph>
0`1TI6@305aLG080I6AT0`2ATI420;jn_P<0fm_K0`3:b/X207EeM@<06ATI0P1KFe/30<K6aP<0KVi^
0P1lO7`307moOdT0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080ooooC01n
OWh207QhN0<0QhN70`2SXj<205eMG@<09BDU0P1^KVh30<o?c`<0lO7a0P3Fe]H30::RXP80J6QX0`0i
>CT307QhN0D000000P0A4A43069RHP<0MWIf0P1PH60304A4A080=SHf0`12@T8306EUI@80TI6A0`2X
Z:P308R8R080:BTY0`00000200h>3P<0I6AT0`1LG5`206ATI0<0TI6A0P2n_[h30=_Kf`<0b/[:0P1e
MGD301TI6@80Fe]K0`36a/H306i^KP80O7al0`1oOgm907inOPD0oooo0@0000010?ooo`003`3oool0
0`000000oooo0?ooo`020?ooodH0OWin0`1oOgl307alO080NWYj0`2IVIT308R8R080<c<c0`19BDT2
0;:b/P<0kNg]0`3Lg=`20:>SX`<0JFUY0P0W9bL304E5A@<0<c<c1@00000204U9B@<0L71`0`1HF5P2
02PX:0P000000`0L71`205IFEP<0Shn?0`2SXj<206a/K0<00P820P00000303De=@<0EUIF0P14A4@3
06=SH`80U9BD0`35aLD30=gMg@80];Bd0`19BDT201lO7`<0YZJV0`2JVYX206m_K`<0P820B@1nOWh5
0?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooom607inOP<0Ogmo0`1lO7`207Yj
NP<0VIVI0`28R8P203<c<`<0BDU90P2b/[830>g]k@<0g=cL0P2SXj<306UYJ@809bLW0`15ADD303<c
<`D000000P19BDT3071`L0<0F5QH0P0X:2P8000000<071`L0P1FEUH308n?S`<0Xj>S0P1/K6`30082
0P8000000`0e=CD305IFEP80A4A40`1SHf<209BDU0<0aLG50`3MgMd20;Bd]0<0BDU90P0O7al30:JV
YP<0VYZJ0P1_Kfl30820P4T0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080
ooooAP1nOWh307moO`<0O7al0P1jNWX309VIV@<0R8R80P0c<c<304U9B@80/[:b0`3]kNd30=cLg080
Xj>S0`1YJFT202LW9`<0ADE50`0c<c<500000080BDU90`1`L70305QHF080:2PX2000000301`L7080
EUIF0`2?Shl30:>SX`80K6a/0`020P82000000<0=CDe0`1FEUH204A4A0<0Hf=S0P2DU9@30<G5a@<0
gMgM0P2d];@304U9B@807alO0`2VYZH309ZJVP80Kfm_0`20P81907inOPD0oooo0@0000010?ooo`00
3`3oool00`000000oooo0?ooo`020?ooodT0OWin0`1iNGT208J6QP<0W9bL0`1HF5P203<c<`<0RXZ:
0P3Ph>030>?Sh`<0Zj^[0P1XJ6P3038b<P803Ph>0`1DE5@5000000<0=CDe0P1nOWh307AdM0<0?Shn
0P0410@;0000008092@T0`1[Jf/30:JVYP80YjNW0`1BDU82000000<01PH60`1<C4`503dm?@80IVIV
0`2KVi/30<o?c`80fMWI0`2@T90202@T90<0E5AD0`2n_[h2071`L0<0OGem0`1oOgm607inOPD0oooo
0@0000010?ooo`003`3oool2000000<0ooooB@1nOWh307UiN@80QXJ60`2LW9`305QHF080<c<c0`2:
RXX20>3Ph0<0hn?S0`2[Zj/206QXJ0<0<S8b0P0>3Ph305ADE0D000000`0e=CD207inOP<0M7Ad0`0n
?Sh200@410/000000P0T92@306][J`<0YZJV0P2WYjL3059BDP8000000`061PH304a<C0D0?Cdm0P1V
IVH309^KV`<0clo?0P3IfMT3092@T08092@T0`1DE5@30;jn_P80L71`0`1mOGd307moOdH0OWin103o
ool200000040oooo000?0?ooo`030000003oool0oooo0080ooooAP1nOWh307alO0<0Ng]k0P2EUID3
0861P@<0=cLg0P1PH6030<O7a`80j>SX0`2h^;P306m_K`80=SHf0`0820P5038b<P<000000P0L71`3
08f=S@80X:2P0`1aLG43034a<@d000000P0820P304U9B@<0UIFE0P36a/H30:NWY`80>CTi0`000003
03Lg=`80BTY:0`0^;Rh203lo?`<0K6a/0`2XZ:P20=WIf@<0bLW90P1OGel301lO7`<0WYjN0P2FUYH3
079bLP<0P820AP1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooom607in
OP<0O7al0`1kNg/209FEU@<0PH610`0g=cL2061PH0<0alO70P3Xj>P30;Rh^0<0Kfm_0P0f=SH300P8
20D0<S8b0`00000201`L70<0SHf=0P2PX:03075aL@<0<C4a3@00000200P820<0BDU90`2EUID20<K6
aP<0YjNW0P0i>CT3000000<0=cLg0P1:BTX302h^;P80?clo0`1/K6`30:RXZ080fMWI0`39bLT205mO
G`<07alO0`2NWYh209JFUP<0LW9b0`20P81607inOPD0oooo0@0000010?ooo`003`3oool00`000000
oooo0?ooo`020?ooodH0OWin0`1lO7`307]kN`80UIFE0`21PH4303Lg=`80H61P0`37alL20>SXj0<0
^;Rh0`1_Kfl203Hf=P<020P81@0b<S830000008071`L0`2=SHd20:2PX0<0LG5a0`0a<C4=00000080
20P80`19BDT309FEU@80a/K60`2WYjL203Ti>@<000000`0g=cL204Y:BP<0;Rh^0P0o?cl306a/K0<0
Z:RX0P3IfMT30<W9b@80GemO0`0O7al309jNWP80UYJF0`1bLW830820P4H0OWin1@3oool100000040
oooo000?0?ooo`030000003oool0oooo0080ooooAP1nOWh307YjNP<0QHF50P2EUID305IFEP<0@T92
0P2QXJ430>CTi080b<S80`1lO7`303Xj>P805ADE0`092@T205mOG`<00@410`051@D208j>SP<0cLg=
0P2^[Zh306i^KP<0=SHf0P0C4a<;000000802@T90`0e=CD307YjNP80a/K60`3Rh^8209^KV`<0:RXZ
0`0T92@205IFEP<0>c/k0P0Z:RX304=3@`<0MWIf0P2i^KT30=kNgP80YjNW0`0`<30304m?C`80]kNg
0`1bLW8307emO@80OgmoA01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3o
oom607inOP<0NWYj0`25QHD209FEU@<0EUIF0`12@T820:6QX@<0i>CT0P38b<P307alO0<0>SXj0P0E
5AD300T92@80GemO0`010@4300D51@80SXj>0`3=cLd20:j^[P<0KVi^0`0f=SH201<C4`/000000P09
2@T303De=@<0NWYj0P36a/H30>;RhP80Vi^K0`0Z:RX302@T9080EUIF0`0k>c/202XZ:P<0@d=30`1f
MWH20;Vi^@<0g]kN0P2WYjL3030`<0<0Cdm?0P2g]kL3079bLP<0OGem0P1oOgm407inOPD0oooo0@00
00010?ooo`003`3oool00`000000oooo0?ooo`020?ooodH0OWin0`1jNWX308F5Q@80UIFE0`1FEUH3
0492@P80XJ6Q0`3Ti>@20<S8b0<0O7al0`0j>SX201DE5@<02@T90P1OGel300410@<01@D50P2>SXh3
0<g=c@80[Zj^0`1^KVh303Hf=P804a<C2`00000200T92@<0=CDe0`1jNWX20<K6aP<0h^;R0P2KVi/3
02XZ:P<092@T0P1FEUH303/k>`80:RXZ0`13@d<307IfMP80^KVi0`3Ng]h20:NWY`<0<30`0`1?Cdl2
0;Ng]`<0LW9b0`1mOGd207moOd@0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo
0080ooooA01nOWh207emO@<0O7al0`2BTY8207alO0<0?Shn0`1dM7@20=?Cd`<0emOG0P2@T90304=3
@`<06QXJ0P092@T304I6AP80FUYJ0`092@T307YjNP80jn_[0`3YjNT20:b/[0<0LG5a0`1?Cdl204A4
A0<0@T920P10@40303Ti>@<0;Rh^0P0X:2P303De=@<0HV9R0P2][Jd30?;blP80k>c/0`23Ph<3038b
<P80EUIF0`1JFUX202l_;`<0;2`/0`1:BTX208J6QP<0bl_;0P3Fe]H307=cL`<08R8R0P2HV9P309:B
TP<0MGEe0P1oOgm407inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oood@0
OWin0P1mOGd307alO0<0TY:B0P1lO7`303hn?P<0M7Ad0P3Cdm<30=OGe`80T92@0`13@d<301XJ6P80
2@T90`16ATH205YJFP<02@T90`1jNWX20>_[j`<0jNWY0P2/[:`3075aL@<0Cdm?0P14A4@30492@P80
@4100`0i>CT302h^;P80:2PX0`0e=CD3069RHP80[Jf]0`3bl_820>c/k0<0Ph>30`0b<S8205IFEP<0
FUYJ0P0_;bl302`/;0<0BTY:0P26QXH30<_;b`80e]KF0`1cLg<3028R8P80V9RH0`2BTY8307EeM@80
OgmoA01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooom407inOP80Ng]k
0`24Q8@3092@T080EeMG0`1@D5030;6a/@80g]kN0`2WYjL205=CD`<07QhN0`0E5AD202DU9@<0RHV9
0P0o?cl305]KF`<0kNg]0P3oool30>?Sh`80XJ6Q0`1lO7`307IfMP80Ph>30`2ATI4209JFUP<0SHf=
0`1gMgL205aLG0<0Bd];0`1FEUH208V9R@<0f][J0P3oool30>3Ph0<0KFe]0P1EEED307MgM`80Cdm?
0`0/;2`302l_;`80EEEE0`2MWId20=[JfP<0^KVi0`0m?Cd204e=C@<0/;2`0`1cLg=607inOPD0oooo
0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oood@0OWin0P1kNg/308B4Q0<0T92@0P1G
EeL3051@D0<0/K6a0P3Ng]h30:NWY`80De=C0`0N7Qh301DE5@809BDU0`29RHT203lo?`<0Fe]K0`3]
kNd20?ooo`<0hn?S0P2QXJ4307alO0<0MWIf0P23Ph<3096AT@80UYJF0`2=SHd307MgM`80G5aL0`1;
Bd/305IFEP80RHV90`3Jf]X20?ooo`<0h>3P0`1]KFd205EEE@<0MgMg0P1?Cdl302`/;0<0;bl_0P1E
EED309fMW@80f][J0`2i^KT303dm?@80CDe=0`2`/;0307=cLdH0OWin1@3oool100000040oooo000?
0?ooo`8000000`3ooom407inOP80Ng]k0`24Q8@3092@T080EeMG0`1@D5030;6a/@80g]kN0`2WYjL2
05=CD`<07QhN0`0E5AD202DU9@<0RHV90P0o?cl305]KF`<0kNg]0P3oool30>?Sh`80XJ6Q0`1lO7`3
07IfMP80Ph>30`2ATI4209JFUP<0SHf=0`1gMgL205aLG0<0Bd];0`1FEUH208V9R@<0f][J0P3oool3
0>3Ph0<0KFe]0P1EEED307MgM`80Cdm?0`0/;2`302l_;`80EEEE0`2MWId20=[JfP<0^KVi0`0m?Cd2
04e=C@<0/;2`0`1cLg=607inOP@0oooo0P0000010?ooo`003`3oool00`000000oooo0?ooo`020?oo
od40OWin1@1mOGd308n?S`<0NGUi0P16ATH308F5Q@<0emOG0P30`<0306YZJP809RHV0`0G5aL302HV
9P80LW9b0`2DU9@2055AD@<0clo?1@3oool30<G5a@80UIFE0`2>SXh30:>SX`80`L710`3IfMT20>?S
h`<0fm_K0`31`L4209ZJVP<0LW9b0`1KFe/206YZJP<0ZZZZ1@3oool30<71`@80JVYZ0`21PH4207al
O0<0ADE50`0/;2`203<c<`<0IVIV0P2g]kL30=cLg0<0QHF50P0W9bL309>CT`<0SXj>0P1gMgL307mo
Od40OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo@@1nOWh507emO@<0
Shn?0`1iNGT204I6AP<0QHF50`3GemL20<30`0<0JVYZ0P0V9RH301LG5`<09RHV0P1bLW8309BDU080
DE5A0`3?cll50?ooo`<0aLG50P2EUID308j>SP<0Xj>S0P31`L430=WIf@80hn?S0`3Kfm/30<71`@80
VYZJ0`1bLW8305]KF`80JVYZ0`2ZZZX50?ooo`<0`L710P1ZJVX30861P@80O7al0`15ADD302`/;080
<c<c0`1VIVH20;Ng]`<0g=cL0`25QHD202LW9`<0Ti>C0`2>SXh207MgM`<0Ogmo@@1nOWh50?ooo`40
00000@3oool000l0oooo00<000000?ooo`3oool00P3ooom107inOP<0O7al0P24Q8@308b<S0<0FEUI
0P1MGEd30;bl_0<0dM7A0P26QXH303De=@805aLG0`0/;2`305ADE080_Kfm0`1kNg/209nOW`H0oooo
0P3Shn<309nOW`80SHf=0`2SXj<30<[:bP80l?3`203oool30?[jnP80d];B0`2OWil306m_K`80FUYJ
0`1gMgL20<S8b0H0oooo0P2OWil30861P@80WIfM0`1bLW830410@080:b/[0`0l?3`207moO`<0clo?
0`36a/H204U9B@<0CDe=0`2ZZZX207EeMD@0OWin1@3oool100000040oooo000?0?ooo`030000003o
ool0oooo0080oooo@@1nOWh307alO080Q8B40`2<S8`305UIF@80GEeM0`2l_;`30=7Ad@80QXJ60`0e
=CD201LG5`<0;2`/0`1DE5@20;fm_@<0Ng]k0P2OWil60?ooo`80hn?S0`2OWil208f=S@<0Xj>S0`3:
b/X20?3`l0P0oooo0`3jn_X20=;BdP<0WinO0`1_Kfl205YJFP<0MgMg0P38b<P60?ooo`80WinO0`21
PH4209fMW@<0LW9b0`10@40202/[:`<0?3`l0P1oOgl30<o?c`<0a/K60P19BDT304e=C@<0ZZZZ0P1e
MGE407inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oood40OWin0`1lO7`2
08B4Q0<0S8b<0`1IFET205eMG@<0_;bl0`3AdM4208J6QP<0=CDe0P0G5aL302`/;0<0E5AD0P2m_Kd3
07]kN`80WinO1P3oool20>?Sh`<0WinO0P2=SHd30:>SX`<0b/[:0P3`l?080?ooo`<0n_[j0P3Bd]83
09nOW`<0Kfm_0P1JFUX307MgM`80b<S81P3oool209nOW`<0PH610P2MWId3079bLP<0@4100P0[:b/3
03`l?080Ogmo0`3?cll30<K6aP80BDU90`1=CDd30:ZZZP80MGEeA01nOWh50?ooo`4000000@3oool0
00l0oooo00<000000?ooo`3oool00P3ooolo07inOPD0OGem0P2<S8`307MgM`<0CDe=0P2ATI430=CD
e0<0YJFU0P1:BTX301/K6`80:BTY0`1BDU830:BTY080a/K60`24Q8@20?gmo@<0oooo0`3lo?`20:BT
Y0<0OGem0P2:RXX30;>c/`<0h>3P3@3oool20?Gem@<0aLG50`2<S8`205]KF`<0D51@0P21PH430>;R
hP<0oooo0P3Ph>03092@T080YJFU0`2RXZ8306MWI`80?3`l0`0Z:RX204];B`<0WinO0`3Kfm/209BD
U0<0;2`/0`2@T90208^;R`<0NGUi0P1oOglo07inOPD0oooo0@0000010?ooo`003`3oool00`000000
oooo0?ooo`020?ooocl0OWin1@1mOGd208b<S0<0MgMg0`1=CDd2096AT@<0e=CD0`2UYJD204Y:BP<0
6a/K0P0Y:BT3059BDP<0Y:BT0P36a/H308B4Q080oOgm0`3oool30?clo080Y:BT0`1mOGd208Z:RP<0
/k>c0`3Ph>0=0?ooo`80mOGe0`35aLD308b<S080Fe]K0`1@D5020861P@<0h^;R0`3oool20>3Ph0<0
T92@0P2UYJD30::RXP<0IfMW0P0l?3`302XZ:P80Bd];0`2OWil30=_Kf`80U9BD0`0/;2`3092@T080
Rh^;0`1iNGT207moOcl0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo
?`1nOWh507emO@80S8b<0`1gMgL304e=C@80TI6A0`3De=@30:FUY@80BTY:0`0K6a/202TY:@<0DU9B
0`2TY:@20<K6aP<0Q8B40P3moOd30?ooo`<0o?cl0P2TY:@307emO@80RXZ:0`2c/k<30>3Ph0d0oooo
0P3emOD30<G5a@<0S8b<0P1KFe/3051@D080PH610`3Rh^830?ooo`80h>3P0`2@T9020:FUY@<0XZ:R
0`1WIfL203`l?0<0:RXZ0P1;Bd/309nOW`<0fm_K0P2DU9@302`/;0<0T92@0P2;Rh/307UiN@80Ogmo
?`1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooolo07inOP80O7al0`23
Ph<208V9R@<0Fe]K0`1XJ6P20<71`@<0`<300`1XJ6P202<S8`<08R8R0P1BDU8308N7Q`<0jNWY0P2Q
XJ430;jn_PD0oooo0`2ZZZX206QXJ0<0IVIV0P2;Rh/30;Vi^@<0hN7Q0P3lo?`;0?ooo`80oOgm0`3K
fm/30:NWY`80K6a/0`0o?cl204E5A@<0T92@0`3emOD20?ooo`<0]KFe0P2MWId30;fm_@<0UiNG0P1O
Gel303De=@80;Bd]0`1TI6@30<30`080c/k>0`1EEED304i>CP80Y:BT0`1fMWI107inOPD0oooo0@00
00010?ooo`0000<0oooo0000000000000P0000030?ooo`8000001@3oool00`000000oooo0?ooo`02
0?ooocl0OWin0P1lO7`308>3P`80RHV90`1KFe/306QXJ080`L710`30`<0306QXJ0808b<S0`0R8R82
059BDP<0QhN70`3YjNT20:6QX@<0_[jn1@3oool30:ZZZP80J6QX0`1VIVH208^;R`<0^KVi0`3QhN42
0?clo0/0oooo0P3moOd30=_Kf`<0YjNW0P1/K6`303lo?`80ADE50`2@T9030?Gem@80oooo0`2e]KD2
09fMW@<0_Kfm0`2GUiL205mOG`<0=CDe0P0];Bd306ATI0<0`<300P3>c/h305EEE@<0CTi>0P2TY:@3
07IfMT40OWin1@3oool100000040oooo00000`3oool000000?ooo`040?ooo`040000003oool0oooo
000000@0oooo00<000000?ooo`3oool00P3ooolo07inOP80OGem0`29RHT207IfMP<0EEEE0`2KVi/2
0<g=c@<0S8b<0`0e=CD201hN7P<0B4Q80P23Ph<30<g=c@<0hn?S0P2DU9@30?gmo@80oooo0`2g]kL3
05MGE`80?Shn0`1LG5`208Z:RP<0/K6a0`3:b/X20=KFeP<0g=cL0P3Ph>030>GUi@<0jNWY0P3Xj>P3
0=WIf@<0^;Rh0P23Ph<304M7A`808b<S0`11@D430:BTY080n_[j0`3MgMd209bLW0<0^KVi0`30`<02
08^;R`<0EUIF0P0^;Rh303Ph>0<0QhN70P3Fe]H30:2PX0<0<S8b0P2>SXh308R8R080NWYj0`1oOgll
07inOPD0oooo0@0000010?ooo`000P3oool00`000000oooo0?ooo`020?ooo`040000003oool0oooo
000000@0oooo0`0000020?ooocl0OWin0P1mOGd308V9R@80MWIf0`1EEED309^KV`80cLg=0`2<S8`3
03De=@807QhN0`18B4P208>3P`<0cLg=0`3Shn<209BDU0<0oOgm0P3oool30;Ng]`<0EeMG0P0n?Sh3
05aLG080RXZ:0`2a/K430<[:bP80e]KF0`3Lg=`20>3Ph0<0iNGU0`3YjNT20>SXj0<0fMWI0`2h^;P2
08>3P`<0AdM70P0S8b<30451@@<0Y:BT0P3jn_X30=gMg@80W9bL0`2i^KT30<30`080Rh^;0`1FEUH2
02h^;P<0>3Ph0`27QhL20=KFeP<0X:2P0`0b<S8208j>SP<0R8R80P1jNWX307moOc`0OWin103oool2
00000040oooo00030?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo0080oooo?`1nOWh207emO@<0RHV90P1fMWH305EEE@<0Vi^K0P3=cLd3
08b<S0<0=CDe0P0N7Qh304Q8B080Ph>30`3=cLd30>?Sh`80U9BD0`3moOd20?ooo`<0]kNg0`1GEeL2
03hn?P<0G5aL0P2:RXX30;6a/@<0b/[:0P3Fe]H30=cLg080h>3P0`3UiND30>WYj@80j>SX0`3IfMT3
0;Rh^080Ph>30`17AdL202<S8`<0@D510`2TY:@20?[jnP<0gMgM0P2LW9`30;Vi^@<0`<300P2;Rh/3
05IFEP80;Rh^0`0h>3P308N7Q`80e]KF0`2PX:03038b<P80SXj>0`28R8P207YjNP<0Ogmo?01nOWh5
0?ooo`4000000@3oool000050?ooo`000000oooo0?ooo`0000000P3oool010000000oooo0?ooo`00
00040?ooo`030000003oool0oooo0080oooo?01nOWh307emO@80PX:20`27QhL205iNGP<0LG5a0`32
`/820:j^[P<0D51@0`0O7al203Xj>P<0O7al0P2b/[830?ooo`<0Z:RX0P2f][H30?ooo`80b/[:0`1C
De<301`L7080;Bd]0`1LG5`208F5Q@<0W9bL0`2QXJ4209fMW@<0V9RH0P2GUiL309jNWP<0Zj^[0P2i
^KT30<71`@<0^KVi0P2GUiL305iNGP808R8R0`0>3Ph304U9B@80^KVi0`3Xj>P20:j^[P<0XJ6Q0`3>
c/h20;Jf]P<0PH610P19BDT302TY:@<0CTi>0P2^[Zh30=?Cd`<0GemO0P1ADE4309nOW`80MgMg?`1n
OWh50?ooo`4000000@3oool00080oooo0P0000040?ooo`8000001@3oool00`000000oooo0?ooo`02
0?oooc`0OWin0`1mOGd208:2PP<0QhN70P1NGUh3075aL@<0`/;20P2^[Zh3051@D0<07alO0P0j>SX3
07alO080/[:b0`3oool30:RXZ080][Jf0`3oool20<[:bP<0De=C0`0L71`202d];@<0G5aL0P25QHD3
09bLW0<0XJ6Q0P2MWId309RHV080UiNG0`2NWYh30:^[Z`80^KVi0`31`L430;Vi^@80UiNG0`1NGUh2
028R8P<03Ph>0`19BDT20;Vi^@<0j>SX0P2^[Zh30:6QX@<0c/k>0P2f][H30861P@80BDU90`0Y:BT3
04i>CP80[Zj^0`3Cdm<305mOG`80DE5A0`2OWil207MgMcl0OWin1@3oool100000040oooo000?0?oo
o`030000003oool0oooo0080oooo?01nOWh307emO@80QhN70`1fMWH205aLG0<0XJ6Q0`33`l<207Ee
M@<0:BTY0`0^;Rh206e]K@<0[:b/0P3Ti>@30>GUi@<0Q8B40P3Lg=`30=[JfP80GemO0`061PH300D5
1@80<c<c0`1RHV8207]kN`<0O7al0`1_Kfl205eMG@<0Cdm?0P1:BTX3055AD@<0Hf=S0P1mOGd309RH
V0<0Z:RX0P2RXZ8307alO080?Cdm0`0410@300H61P80GUiN0`34a<@20<;2`P<0TI6A0`2i^KT20=3@
d0<0Zj^[0P1eMGD303/k>`<0;Bd]0P1`L7030<k>cP<0ZJVY0P0h>3P308b<S080QHF50`1kNg/307mo
OcT0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo?01nOWh307emO@80
QhN70`1fMWH205aLG0<0XJ6Q0`33`l<207EeM@<0:BTY0`0^;Rh206e]K@<0[:b/0P3Ti>@30>GUi@<0
Q8B40P3Lg=`30=[JfP80GemO0`061PH300D51@80<c<c0`1RHV8207]kN`<0O7al0`1_Kfl205eMG@<0
Cdm?0P1:BTX3055AD@<0Hf=S0P1mOGd309RHV0<0Z:RX0P2RXZ8307alO080?Cdm0`0410@300H61P80
GUiN0`34a<@20<;2`P<0TI6A0`2i^KT20=3@d0<0Zj^[0P1eMGD303/k>`<0;Bd]0P1`L7030<k>cP<0
ZJVY0P0h>3P308b<S080QHF50`1kNg/307moOcT0OWin1@3oool100000040oooo000?0?ooo`030000
003oool0oooo0080oooo?01nOWh307emO@80QhN70`1fMWH205aLG0<0XJ6Q0`33`l<207EeM@<0:BTY
0`0^;Rh206e]K@<0[:b/0P3Ti>@30>GUi@<0Q8B40P3Lg=`30=[JfP80GemO0`061PH300D51@80<c<c
0`1RHV8207]kN`<0O7al0`1_Kfl205eMG@<0Cdm?0P1:BTX3055AD@<0Hf=S0P1mOGd309RHV0<0Z:RX
0P2RXZ8307alO080?Cdm0`0410@300H61P80GUiN0`34a<@20<;2`P<0TI6A0`2i^KT20=3@d0<0Zj^[
0P1eMGD303/k>`<0;Bd]0P1`L7030<k>cP<0ZJVY0P0h>3P308b<S080QHF50`1kNg/307moOcT0OWin
1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo>@1nOWh307emO@<0PH610P25
QHD3069RHP80NGUi0`30`<0309^KV`80?clo0`0U9BD305UIF@80Xj>S0`3>c/h20?gmo@<0UIFE0`2D
U9@20=GEe@<0NGUi0P030`<3000000<0410@0P18B4P306UYJ@80K6a/0`1JFUX303lo?`809BDU0`0B
4Q8200/;2`<0410@0`0S8b<20492@P<0IfMW0`2:RXX209jNWP<0U9BD0P1VIVH301lO7`<000000P0>
3Ph307UiN@80_;bl0`2DU9@309:BTP80c<c<0`39bLT20:6QX@<0I6AT0`0_;bl203dm?@<0W9bL0`3D
e=@206MWI`<0E5AD0P2KVi/307QhN3`0OWin1@3oool100000040oooo000?0?ooo`030000003oool0
oooo0080oooo>@1nOWh307emO@<0PH610P25QHD3069RHP80NGUi0`30`<0309^KV`80?clo0`0U9BD3
05UIF@80Xj>S0`3>c/h20?gmo@<0UIFE0`2DU9@20=GEe@<0NGUi0P030`<3000000<0410@0P18B4P3
06UYJ@80K6a/0`1JFUX303lo?`809BDU0`0B4Q8200/;2`<0410@0`0S8b<20492@P<0IfMW0`2:RXX2
09jNWP<0U9BD0P1VIVH301lO7`<000000P0>3Ph307UiN@80_;bl0`2DU9@309:BTP80c<c<0`39bLT2
0:6QX@<0I6AT0`0_;bl203dm?@<0W9bL0`3De=@206MWI`<0E5AD0P2KVi/307QhN3`0OWin1@3oool1
00000040oooo000?0?ooo`030000003oool0oooo0080oooo>@1nOWh307emO@<0PH610P25QHD3069R
HP80NGUi0`30`<0309^KV`80?clo0`0U9BD305UIF@80Xj>S0`3>c/h20?gmo@<0UIFE0`2DU9@20=GE
e@<0NGUi0P030`<3000000<0410@0P18B4P306UYJ@80K6a/0`1JFUX303lo?`809BDU0`0B4Q8200/;
2`<0410@0`0S8b<20492@P<0IfMW0`2:RXX209jNWP<0U9BD0P1VIVH301lO7`<000000P0>3Ph307Ui
N@80_;bl0`2DU9@309:BTP80c<c<0`39bLT20:6QX@<0I6AT0`0_;bl203dm?@<0W9bL0`3De=@206MW
I`<0E5AD0P2KVi/307QhN3`0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080
oooo?01nOWh308J6QP80MWIf0`1RHV820:BTY0<0]kNg0`1RHV8202HV9P<0ADE50`2ATI420<[:bP<0
i^KV0P3?cll3061PH0<0[Jf]0P2FUYH301DE5@D000000`0e=CD206ATI0<0Kfm_0P1MGEd303hn?P<0
7AdM0P010@4;0000008051@D0`0j>SX306IVIP80SHf=0`2PX:0208j>SP<0DU9B0`051@D2000000<0
9RHV0P2?Shl309fMW@<0MWIf0P2WYjL30=7Ad@80`L710`2CTi<304m?C`80:BTY0`1LG5`30<C4a080
/;2`0`0n?Sh208^;R`<0Ph>30`1lO7`i07inOPD0oooo0@0000010?ooo`003`3oool2000000<0oooo
?01nOWh308J6QP80MWIf0`1RHV820:BTY0<0]kNg0`1RHV8202HV9P<0ADE50`2ATI420<[:bP<0i^KV
0P3?cll3061PH0<0[Jf]0P2FUYH301DE5@D000000`0e=CD206ATI0<0Kfm_0P1MGEd303hn?P<07AdM
0P010@4;0000008051@D0`0j>SX306IVIP80SHf=0`2PX:0208j>SP<0DU9B0`051@D2000000<09RHV
0P2?Shl309fMW@<0MWIf0P2WYjL30=7Ad@80`L710`2CTi<304m?C`80:BTY0`1LG5`30<C4a080/;2`
0`0n?Sh208^;R`<0Ph>30`1lO7`i07inOP@0oooo0P0000010?ooo`003`3oool00`000000oooo0?oo
o`020?ooocL0OWin0P1mOGd30861P@<0Q8B40P1UIFD307moO`80_;bl0`2:RXX303De=@80<c<c0`1j
NWX30<71`@80emOG0`3WinL2071`L0<0J6QX0`2OWil203Xj>PD000000`0U9BD306EUI@80OGem0`1b
LW82059BDP<0;bl_0`0@410?000000<07AdM0`17AdL207EeM@<0WIfM0P2ZZZX308^;R`<0?clo1@00
000204Y:BP<0TY:B0`1cLg<207IfMP<0^k^k0P3=cLd30;Vi^@<0P8200P0k>c/3034a<@<0RXZ:0P3C
dm<306i^KP80EeMG0`2GUiL307UiNCT0OWin1@3oool100000040oooo000?0?ooo`030000003oool0
oooo0080oooo=`1nOWh207emO@<0PH610`24Q8@206EUI@<0Ogmo0P2l_;`308Z:RP<0=CDe0P0c<c<3
07YjNP<0`L710P3GemL30>OWi`80L71`0`1XJ6P309nOW`80>SXj1@00000302DU9@<0IFEU0P1mOGd3
079bLP80DU9B0`0_;bl3010@40l000000`0M7Ad304M7A`80MGEe0`2MWId20:ZZZP<0Rh^;0`0o?cl5
00000080BTY:0`2BTY8307=cL`80MWIf0`2k^k/20<g=c@<0^KVi0`20P80203/k>`<0<C4a0`2:RXX2
0=?Cd`<0KVi^0P1GEeL309NGU`<0NGUi>@1nOWh50?ooo`4000000@3oool000l0oooo00<000000?oo
o`3oool00P3ooolg07inOP80OGem0`21PH4308B4Q080IFEU0`1oOgl20;bl_0<0RXZ:0`0e=CD203<c
<`<0NWYj0`31`L420=OGe`<0inOW0P1`L70306QXJ0<0WinO0P0j>SX5000000<09BDU0`1UIFD207em
O@<0LW9b0P1BDU8302l_;`<0410@3`00000301dM7@<0AdM70P1eMGD309fMW@80ZZZZ0`2;Rh/303lo
?`D000000P1:BTX309:BTP<0Lg=c0P1fMWH30;^k^`80cLg=0`2i^KT30820P080>c/k0`0a<C4308Z:
RP80dm?C0`1^KVh205MGE`<0UiNG0`1iNGTi07inOPD0oooo0@0000010?ooo`003`3oool00`000000
oooo0?ooo`020?ooocT0OWin0`24Q8@307IfMP80IfMW0`2UYJD20:^[Z`<0De=C0`0[:b/2061PH0<0
[jn_0`3GemL20=KFeP<0ZZZZ0P0f=SH308:2PP<0I6AT1@00000201DE5@<0IfMW0`2ATI42092@T0<0
Lg=c0P1@D503034a<@<071`L0P0>3Ph300H61P800@411P00000200H61P<05aLG0`0d=3@205iNGP<0
SXj>0P2d];@30;Ng]`<0QHF50P0/;2`30000008030`<0`1ZJVX307alO080EeMG0`2;Rh/20<?3``<0
b/[:0`2][Jd206MWI`<0;Bd]0`1<C4`20;Vi^@<0];Bd0P14A4@308Z:RP<0PH610P1mOGdg07inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooocT0OWin0`24Q8@307IfMP80IfMW
0`2UYJD20:^[Z`<0De=C0`0[:b/2061PH0<0[jn_0`3GemL20=KFeP<0ZZZZ0P0f=SH308:2PP<0I6AT
1@00000201DE5@<0IfMW0`2ATI42092@T0<0Lg=c0P1@D503034a<@<071`L0P0>3Ph300H61P800@41
1P00000200H61P<05aLG0`0d=3@205iNGP<0SXj>0P2d];@30;Ng]`<0QHF50P0/;2`30000008030`<
0`1ZJVX307alO080EeMG0`2;Rh/20<?3``<0b/[:0`2][Jd206MWI`<0;Bd]0`1<C4`20;Vi^@<0];Bd
0P14A4@308Z:RP<0PH610P1mOGdg07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?oo
o`020?ooocT0OWin0`24Q8@307IfMP80IfMW0`2UYJD20:^[Z`<0De=C0`0[:b/2061PH0<0[jn_0`3G
emL20=KFeP<0ZZZZ0P0f=SH308:2PP<0I6AT1@00000201DE5@<0IfMW0`2ATI42092@T0<0Lg=c0P1@
D503034a<@<071`L0P0>3Ph300H61P800@411P00000200H61P<05aLG0`0d=3@205iNGP<0SXj>0P2d
];@30;Ng]`<0QHF50P0/;2`30000008030`<0`1ZJVX307alO080EeMG0`2;Rh/20<?3``<0b/[:0`2]
[Jd206MWI`<0;Bd]0`1<C4`20;Vi^@<0];Bd0P14A4@308Z:RP<0PH610P1mOGdg07inOPD0oooo0@00
00010?ooo`003`3oool00`000000oooo0?ooo`020?ooocL0OWin0P20P80308:2PP<0J6QX0P23Ph<3
0;Jf]P80Ng]k0`0a<C4304M7A`80UiNG0`3Bd]830<o?c`80`/;20`17AdL20492@P<0O7al0`0L71`2
000000<01PH60P1UIFD30:FUY@<0/K6a0P2IVIT307IfMP80F5QH0`16ATH303hn?PD0?3`l0P0j>SX3
03De=@<0;bl_1@0Y:BT303@d=080Cdm?0`1jNWX20:^[Z`<0cLg=0`31`L4207]kN`<06a/K0P000003
03@d=0<0MgMg0P1HF5P305MGE`80WinO0`35aLD30<O7a`80VYZJ0`1<C4`302`/;080NWYj0`3@d=02
07=cL`<0Fe]K0`2CTi<207YjNSL0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo
0080oooo=`1nOWh20820P0<0PX:20`1XJ6P208>3P`<0][Jf0P1kNg/3034a<@<0AdM70P2GUiL30=;B
dP<0clo?0P32`/8304M7A`80@T920`1lO7`301`L708000000`061PH206EUI@<0YJFU0`2a/K4209VI
V@<0MWIf0P1HF5P304I6AP<0?Shn1@0l?3`203Xj>P<0=CDe0`0_;bl502TY:@<0=3@d0P1?Cdl307Yj
NP80Zj^[0`3=cLd30<71`@80Ng]k0`0K6a/2000000<0=3@d0`1gMgL205QHF0<0EeMG0P2OWil30<G5
a@<0alO70P2JVYX304a<C0<0;2`/0P1jNWX30=3@d080Lg=c0`1KFe/309>CT`80NWYj=`1nOWh50?oo
o`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooolg07inOP80Ph>30`1gMgL306][J`80
YJFU0`2PX:0204U9B@<0=CDe0`1kNg/20<G5a@<0e]KF0`2h^;P20820P0<04A4A0P1XJ6P3051@D0D0
00000`1LG5`20;:b/P<0d=3@0`2n_[h209ZJVP<0Ng]k0P1[Jf/306QXJ0<0KVi^0P1hN7P307moO`80
P8200`1kNg/306i^KP80GUiN0`1?Cdl304I6AP80CDe=0`1XJ6P209JFUP<0b<S80`3Shn<20<C4a0<0
Jf][0P0@410300h>3P<0FUYJ0P1XJ6P303hn?P80Jf][0`2[Zj/30<S8b080_kno0`20P80303De=@80
@4100`2][Jd20;Jf]P<0BDU90`2:RXX20820P0<0OGem=01nOWh50?ooo`4000000@3oool000l0oooo
00<000000?ooo`3oool00P3ooolg07inOP80Ph>30`1gMgL306][J`80YJFU0`2PX:0204U9B@<0=CDe
0`1kNg/20<G5a@<0e]KF0`2h^;P20820P0<04A4A0P1XJ6P3051@D0D000000`1LG5`20;:b/P<0d=3@
0`2n_[h209ZJVP<0Ng]k0P1[Jf/306QXJ0<0KVi^0P1hN7P307moO`80P8200`1kNg/306i^KP80GUiN
0`1?Cdl304I6AP80CDe=0`1XJ6P209JFUP<0b<S80`3Shn<20<C4a0<0Jf][0P0@410300h>3P<0FUYJ
0P1XJ6P303hn?P80Jf][0`2[Zj/30<S8b080_kno0`20P80303De=@80@4100`2][Jd20;Jf]P<0BDU9
0`2:RXX20820P0<0OGem=01nOWh50?ooo`4000000@3oool000l0oooo0P0000030?ooocL0OWin0P23
Ph<307MgM`<0Jf][0P2UYJD30:2PX080BDU90`0e=CD307]kN`80aLG50`3Fe]H30;Rh^080P8200`0A
4A4206QXJ0<0D51@1@00000305aLG080/[:b0`3@d=030;jn_P80VYZJ0`1kNg/206][J`<0J6QX0`1^
KVh207QhN0<0Ogmo0P20P80307]kN`<0KVi^0P1NGUh304m?C`<0ATI60P1=CDd306QXJ080UYJF0`38
b<P30>?Sh`80a<C40`1[Jf/2010@40<03Ph>0`1JFUX206QXJ0<0?Shn0P1[Jf/30:^[Z`<0b<S80P2o
_kl30820P0<0=CDe0P10@4030:f][@80][Jf0`19BDT308Z:RP80P8200`1mOGdd07inOP@0oooo0P00
00010?ooo`003`3oool00`000000oooo0?ooo`020?oooc@0OWin0`20P80208:2PP<0Jf][0`26QXH2
0;2`/0<0Kfm_0P0b<S8305aLG0<0/;2`0P3GemL30;Zj^P<0V9RH0P0T92@302`/;080LW9b0`0N7Qh3
00000080C4a<0`2f][H20>WYj@<0h>3P0`2j^[X209NGU`<0QHF50P24Q8@3096AT@<0Xj>S0P2e]KD3
0<71`@80aLG50`2o_kl30;6a/@80VYZJ0`20P80306QXJ080FUYJ0`1OGel207emO@<0/;2`0`3Shn<2
0?;blP<0_[jn0P1GEeL301<C4`<0=cLg0P1_Kfl304Q8B080?Cdm0`1oOgl30;:b/P80bl_;0`2_[jl3
061PH080:b/[0`1/K6`20<_;b`<0MgMg0`1OGel2092@T0<0Ng]k=01nOWh50?ooo`4000000@3oool0
00l0oooo00<000000?ooo`3oool00P3ooold07inOP<0P8200P22PX8306][J`<0QXJ60P2`/;0306m_
K`80<S8b0`1LG5`30;2`/080emOG0`2j^[X309RHV08092@T0`0/;2`2079bLP<07QhN0`00000204a<
C0<0][Jf0P3YjNT30>3Ph0<0^[Zj0P2GUiL308F5Q@80Q8B40`2ATI430:>SX`80]KFe0`31`L420<G5
a@<0_kno0`2a/K4209ZJVP<0P8200`1XJ6P205YJFP<0GemO0P1mOGd30;2`/0<0hn?S0P3bl_830;jn
_P80EeMG0`0C4a<303Lg=`80Kfm_0`18B4P203dm?@<0Ogmo0`2b/[820<_;b`<0[jn_0`1PH60202/[
:`<0K6a/0P3;bl/307MgM`<0GemO0P2@T90307]kNc@0OWin1@3oool100000040oooo000?0?ooo`03
0000003oool0oooo0080oooo=01nOWh308:2PP80N7Qh0`1_Kfl30:BTY080UIFE1@13@d<309>CT`<0
d=3@0P38b<P309BDU0<0FUYJ0P00000306EUI@80F5QH0`0>3Ph303/k>`80[:b/0`3hn?P20?clo0<0
e]KF0`2[Zj/209:BTP<0T92@0P2OWil30;Ng]`<0d=3@0P3UiND30??cl`80nOWi0`3emOD30>OWi`80
d=3@0`2a/K43092@T080LW9b0`1QHF4206QXJ0<0SHf=0`37alL20?[jnP<0n?Sh0P2][Jd304E5A@<0
9bLW0P1PH60306IVIP80;bl_0`1>CTh308f=S@80^k^k0`39bLT309JFUP80@T920`0h>3P20::RXP<0
]kNg0`1=CDd208V9R@<0Ogmo=01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool0
0P3ooold07inOP<0PX:20P1hN7P306m_K`<0Y:BT0P2EUID504=3@`<0Ti>C0`3@d=020<S8b0<0U9BD
0`1JFUX2000000<0IFEU0P1HF5P300h>3P<0>c/k0P2/[:`30?Shn080o?cl0`3Fe]H30:^[Z`80TY:B
0`2@T90209nOW`<0]kNg0`3@d=020>GUi@<0lo?c0P3inOT30?Gem@<0inOW0P3@d=030;6a/@<0T92@
0P1bLW83065QH@80J6QX0`2=SHd30<O7a`80n_[j0`3hn?P20:f][@<0ADE50`0W9bL2061PH0<0IVIV
0P0_;bl304i>CP<0SHf=0P2k^k/30<W9b@<0UYJF0P12@T8303Ph>080XZ:R0`2g]kL304e=C@80RHV9
0`1oOgld07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooc@0OWin0`22
PX8207QhN0<0Kfm_0`2TY:@209FEU@D0@d=30`2CTi<30=3@d080b<S80`2DU9@305YJFP8000000`1U
IFD205QHF0<03Ph>0`0k>c/20:b/[0<0n?Sh0P3lo?`30=KFeP<0Zj^[0P2BTY83092@T080WinO0`2g
]kL30=3@d080iNGU0`3clo<20?Win@<0mOGe0`3WinL20=3@d0<0/K6a0`2@T902079bLP<0HF5Q0P1X
J6P308f=S@<0alO70P3jn_X30?Shn080[Jf]0`15ADD302LW9`80H61P0`1VIVH202l_;`<0CTi>0`2=
SHd20;^k^`<0bLW90`2FUYH20492@P<0>3Ph0P2RXZ830;Ng]`<0CDe=0P29RHT307moOc@0OWin1@3o
ool100000040oooo000?0?ooo`030000003oool0oooo0080oooo<P1nOWh207moO`<0PH610P1^KVh3
08N7Q`<0ZZZZ0P1VIVH303Lg=`80LG5a0`31`L430=7Ad@80WYjN0`1`L70300h>3P80:RXZ0`20P802
03Ti>@<0<C4a0`2GUiL20?[jnP<0oooo0P3]kNd30;Vi^@<0UYJF0P2>SXh309^KV`80]KFe0`3AdM43
0>_[j`d0oooo0P3clo<30=OGe`<0]KFe0P2?Shl306m_K`80GemO0`1/K6`309^KV`80gMgM0`3oool2
0?;blP<0U9BD0`0o?cl204a<C0<0N7Qh0P19BDT302`/;0<0HV9R0P2IVIT30<G5a@<0_kno0P1eMGD3
02l_;`80H61P0`36a/H307YjNP80Hf=S0`2=SHd207]kNc80OWin1@3oool100000040oooo000?0?oo
o`030000003oool0oooo0080oooo<P1nOWh207moO`<0PH610P1^KVh308N7Q`<0ZZZZ0P1VIVH303Lg
=`80LG5a0`31`L430=7Ad@80WYjN0`1`L70300h>3P80:RXZ0`20P80203Ti>@<0<C4a0`2GUiL20?[j
nP<0oooo0P3]kNd30;Vi^@<0UYJF0P2>SXh309^KV`80]KFe0`3AdM430>_[j`d0oooo0P3clo<30=OG
e`<0]KFe0P2?Shl306m_K`80GemO0`1/K6`309^KV`80gMgM0`3oool20?;blP<0U9BD0`0o?cl204a<
C0<0N7Qh0P19BDT302`/;0<0HV9R0P2IVIT30<G5a@<0_kno0P1eMGD302l_;`80H61P0`36a/H307Yj
NP80Hf=S0`2=SHd207]kNc80OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080
oooo<P1nOWh207moO`<0PH610P1^KVh308N7Q`<0ZZZZ0P1VIVH303Lg=`80LG5a0`31`L430=7Ad@80
WYjN0`1`L70300h>3P80:RXZ0`20P80203Ti>@<0<C4a0`2GUiL20?[jnP<0oooo0P3]kNd30;Vi^@<0
UYJF0P2>SXh309^KV`80]KFe0`3AdM430>_[j`d0oooo0P3clo<30=OGe`<0]KFe0P2?Shl306m_K`80
GemO0`1/K6`309^KV`80gMgM0`3oool20?;blP<0U9BD0`0o?cl204a<C0<0N7Qh0P19BDT302`/;0<0
HV9R0P2IVIT30<G5a@<0_kno0P1eMGD302l_;`80H61P0`36a/H307YjNP80Hf=S0`2=SHd207]kNc80
OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo<P1nOWh208:2PP<0N7Qh
0P1bLW830::RXP<0S8b<0P11@D43059BDP80YjNW0`3Bd]830;:b/P80L71`0`0o?cl300000080MWIf
0`1bLW8203Xj>P<0O7al0`3]kNd50?ooo`80a<C40`2CTi<308:2PP80Rh^;0`2SXj<20<30`0<0f][J
0`3^k^h20?clo0/0oooo0P3lo?`30>WYj@<0c/k>0P2ZZZX308>3P`80HV9R0`1FEUH306e]K@80Z:RX
0`3_knl20?ooo`<0gmoO0`1jNWX204Y:BP<0M7Ad0P1cLg<302l_;`<0>SXj0P1`L7030:RXZ0<0bl_;
0P2ZZZX3055AD@80<c<c0`2HV9P30;Ng]`80DE5A0`29RHTd07inOPD0oooo0@0000010?ooo`003`3o
ool00`000000oooo0?ooo`020?oooc80OWin0P22PX8307QhN080LW9b0`2RXZ8308b<S080@D510`1B
DU820:NWY`<0d];B0`2b/[82071`L0<0?clo0`00000207IfMP<0LW9b0P0j>SX307alO0<0kNg]1@3o
ool20<C4a0<0Ti>C0`22PX8208^;R`<0Xj>S0P30`<030=[JfP<0k^k^0P3lo?`;0?ooo`80o?cl0`3Y
jNT30<k>cP80ZZZZ0`23Ph<2069RHP<0EUIF0`1]KFd20:RXZ0<0kno_0P3oool30=oOg`<0NWYj0P1:
BTX307AdM080Lg=c0`0_;bl303Xj>P80L71`0`2XZ:P30<_;b`80ZZZZ0`1ADE4203<c<`<0V9RH0`2g
]kL2055AD@<0RHV9=01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3oool_
07inOP<0Ogmo0P20P803071`L080RHV90`2SXj<305mOG`80?clo0`25QHD20<[:bP<0`/;20`1oOgl2
055AD@<02@T90`0k>c/209jNWP<0GEeM0P1VIVH30=;BdPD0oooo0`3>c/h208n?S`<0L71`0`1bLW82
08R8R0<0Y:BT0P2l_;`30<k>cP<0fMWI0P3Ng]h30>;RhP80iNGU0`3WinL50>WYj@<0hn?S0`3De=@2
0;^k^`<0UiNG0P1_Kfl304m?C`<0BTY:0P1]KFd30;Jf]P80o_kn0`3oool30<71`@D0J6QX0P2>SXh3
05MGE`<0:BTY0P1;Bd/307inOP<0^;Rh0P38b<P308V9R@80=CDe0`1FEUH30<71`@80O7al0`1WIfL2
08Z:RP<0O7al;`1nOWh50?ooo`4000000@3oool000l0oooo0P0000030?ooobl0OWin0`1oOgl20820
P0<0L71`0P29RHT30:>SX`<0GemO0P0o?cl308F5Q@80b/[:0`32`/8307moO`80DE5A0`092@T303/k
>`80WYjN0`1MGEd206IVIP<0d];B1@3oool30<k>cP80Shn?0`1`L703079bLP80R8R80`2TY:@20;bl
_0<0c/k>0`3IfMT20=kNgP<0h^;R0P3UiND30>OWi`D0jNWY0`3Shn<30=CDe080^k^k0`2GUiL206m_
K`<0Cdm?0`1:BTX206e]K@<0][Jf0P3no_h30?ooo`<0`L711@1XJ6P208j>SP<0EeMG0`0Y:BT204];
B`<0OWin0`2h^;P20<S8b0<0RHV90P0e=CD305IFEP<0`L710P1lO7`306MWI`80RXZ:0`1lO7`_07in
OP@0oooo0P0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooobl0OWin0`1oOgl20820P0<0
L71`0P29RHT30:>SX`<0GemO0P0o?cl308F5Q@80b/[:0`32`/8307moO`80DE5A0`092@T303/k>`80
WYjN0`1MGEd206IVIP<0d];B1@3oool30<k>cP80Shn?0`1`L703079bLP80R8R80`2TY:@20;bl_0<0
c/k>0`3IfMT20=kNgP<0h^;R0P3UiND30>OWi`D0jNWY0`3Shn<30=CDe080^k^k0`2GUiL206m_K`<0
Cdm?0`1:BTX206e]K@<0][Jf0P3no_h30?ooo`<0`L711@1XJ6P208j>SP<0EeMG0`0Y:BT204];B`<0
OWin0`2h^;P20<S8b0<0RHV90P0e=CD305IFEP<0`L710P1lO7`306MWI`80RXZ:0`1lO7`_07inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooobl0OWin0`21PH4207UiN@<0M7Ad
0P2OWil308F5Q@<0@D510P1RHV830;Fe]@80c<c<0`2HV9P3059BDP80<S8b0`092@T309JFUP80U9BD
0`1SHf<20:j^[PH0oooo0P3IfMT308b<S080GEeM0`1FEUH306UYJ@80Q8B40`2LW9`20:ZZZP<0/;2`
0`2_[jl20:b/[0D0ZZZZ0`2^[Zh30;Jf]P80_kno0`36a/H30<O7a`80_Kfm0`2UYJD20861P@<0F5QH
0`0j>SX203hn?P<0LG5a0P35aLD30?ooo`<0moOg0P2PX:0306YZJP80SXj>0`2:RXX303/k>`80<S8b
0`1GEeL3092@T080a/K60`2i^KT2065QH@<0<30`0`2?Shl20;Jf]P<0EEEE0P29RHTb07inOPD0oooo
0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooobl0OWin0`21PH4207UiN@<0M7Ad0P2O
Wil308F5Q@<0@D510P1RHV830;Fe]@80c<c<0`2HV9P3059BDP80<S8b0`092@T309JFUP80U9BD0`1S
Hf<20:j^[PH0oooo0P3IfMT308b<S080GEeM0`1FEUH306UYJ@80Q8B40`2LW9`20:ZZZP<0/;2`0`2_
[jl20:b/[0D0ZZZZ0`2^[Zh30;Jf]P80_kno0`36a/H30<O7a`80_Kfm0`2UYJD20861P@<0F5QH0`0j
>SX203hn?P<0LG5a0P35aLD30?ooo`<0moOg0P2PX:0306YZJP80SXj>0`2:RXX303/k>`80<S8b0`1G
EeL3092@T080a/K60`2i^KT2065QH@<0<30`0`2?Shl20;Jf]P<0EEEE0P29RHTb07inOPD0oooo0@00
00010?ooo`003`3oool00`000000oooo0?ooo`020?ooob`0OWin0`1oOgl30820P080LW9b0`29RHT2
09jNWP<0FUYJ0`18B4P209JFUP<0c<c<0P2^[Zh3069RHP<0?Cdm0P0C4a<305UIF@<0`L710P1nOWh3
08b<S080kno_0`3oool30>GUi@80SHf=0`1>CTh203/k>`<0BTY:0`1VIVH207inOP<0S8b<0P2=SHd3
08J6QP<0Ng]k0P1`L70306UYJ@80IfMW0`1/K6`307MgM`80QhN70`2HV9P30:RXZ080[jn_0`2XZ:P2
096AT@<0Jf][0`10@40202HV9P<0=SHf0P1jNWX30=CDe0<0oooo0P3Hf=P308J6QP80PH610`2WYjL3
06m_K`80<C4a0`0l?3`306EUI@80YZJV0`3;bl/209ZJVP<0?Shn0`1>CTh20;bl_0<0OGem0P1ZJVX3
08R8R0<0O7al;01nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3oool/07in
OP<0Ogmo0`20P802079bLP<0RHV90P2NWYh305YJFP<0B4Q80P2FUYH30<c<c080[Zj^0`1RHV8303dm
?@804a<C0`1IFET30<71`@80OWin0`2<S8`20>o_k`<0oooo0`3UiND208f=S@<0CTi>0P0k>c/304Y:
BP<0IVIV0P1nOWh308b<S080SHf=0`26QXH307]kN`80L71`0`1YJFT206MWI`<0K6a/0`1gMgL208N7
Q`<0V9RH0`2XZ:P20:n_[`<0Z:RX0P2ATI4306][J`<0@4100P0V9RH303Hf=P80NWYj0`3De=@30?oo
o`80f=SH0`26QXH20861P@<0YjNW0`1_Kfl2034a<@<0?3`l0`1UIFD20:JVYP<0bl_;0P2JVYX303hn
?P<0CTi>0P2l_;`307emO@80JVYZ0`28R8P307alO2`0OWin1@3oool100000040oooo000?0?ooo`03
0000003oool0oooo0080oooo;01nOWh307moO`<0P8200P1bLW8308V9R@80WYjN0`1JFUX304Q8B080
UYJF0`3<c<`20:j^[P<0HV9R0`0m?Cd201<C4`<0FEUI0`31`L4207inOP<0S8b<0P3_knl30?ooo`<0
iNGU0P2=SHd304i>CP80>c/k0`1:BTX306IVIP80OWin0`2<S8`208f=S@<0QXJ60`1kNg/2071`L0<0
JFUY0P1WIfL306a/K0<0MgMg0P27QhL309RHV0<0Z:RX0P2_[jl30:RXZ080TI6A0`1[Jf/30410@080
9RHV0`0f=SH207YjNP<0e=CD0`3oool20=SHf0<0QXJ60P21PH430:NWY`<0Kfm_0P0a<C4303`l?0<0
IFEU0P2VYZH30<_;b`80VYZJ0`0n?Sh304i>CP80_;bl0`1mOGd206YZJP<0R8R80`1lO7`/07inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooob`0OWin0`20P80307YjNP80MWIf
0`2MWId207inOP<0A4A40`1aLG420;no_`<0`<300P1mOGd303`l?0<0<c<c0P0U9BD30;Zj^P<0/[:b
0P1nOWh30<G5a@80oooo0`3`l?0309FEU@80A4A40`0S8b<202h^;P<0C4a<0`1XJ6P507IfMP80Jf][
0`1JFUX304Q8B080>3Ph0`0^;Rh202/[:`<0;bl_0`0k>c/204i>CP<0IFEU0`1nOWh209BDU0<0WinO
0P2JVYX30861P@<0EUIF0P0Y:BT301DE5@80=3@d0`28R8P30>3Ph080m?Cd0`2b/[8207moO`<0Y:BT
0`2UYJD2059BDP<0=3@d0`15ADD207QhN0<0^k^k0P33`l<3075aL@<0<C4a0P27QhL30;Bd]080FEUI
0`28R8P307emOB`0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo;01n
OWh30820P0<0NWYj0P1fMWH309fMW@80OWin0`14A4@3075aL@80_kno0`30`<0207emO@<0?3`l0`0c
<c<202DU9@<0^[Zj0`2b/[8207inOP<0aLG50P3oool30?3`l0<0UIFE0P14A4@302<S8`80;Rh^0`1<
C4`306QXJ0D0MWIf0P1[Jf/305YJFP<0B4Q80P0h>3P302h^;P80:b/[0`0_;bl303/k>`80CTi>0`1U
IFD307inOP80U9BD0`2OWil209ZJVP<0PH610`1FEUH202TY:@<05ADE0P0d=3@308R8R0<0h>3P0P3d
m?@30;:b/P80Ogmo0`2TY:@30:FUY@80DU9B0`0d=3@304E5A@80N7Qh0`2k^k/20<?3``<0LG5a0`0a
<C4208N7Q`<0];Bd0P1IFET308R8R0<0OGem;01nOWh50?ooo`4000000@3oool000l0oooo00<00000
0?ooo`3oool00P3oool/07inOP<0P8200`1jNWX207IfMP<0WIfM0P1nOWh304A4A0<0LG5a0P2o_kl3
0<30`080OGem0`0l?3`303<c<`809BDU0`2j^[X30;:b/P80OWin0`35aLD20?ooo`<0l?3`0`2EUID2
04A4A0<08b<S0P0^;Rh304a<C0<0J6QX1@1fMWH206][J`<0FUYJ0`18B4P203Ph>0<0;Rh^0P0[:b/3
02l_;`<0>c/k0P1>CTh306EUI@<0OWin0P2DU9@309nOW`80VYZJ0`21PH4305IFEP80:BTY0`0E5AD2
03@d=0<0R8R80`3Ph>020?Cdm0<0/[:b0P1oOgl30:BTY0<0YJFU0P1BDU8303@d=0<0ADE50P1hN7P3
0;^k^`80`l?30`1aLG43034a<@80QhN70`2d];@205UIF@<0R8R80`1mOGd/07inOPD0oooo0@000001
0?ooo`003`3oool00`000000oooo0?ooo`020?ooobX0OWin0P1oOgl30820P0<0M7Ad0P29RHT309RH
V080F5QH0`1CDe<30:>SX`80bLW90`2GUiL204];B`<0=CDe0`0Z:RX207emO@<0hN7Q0`2ATI4209bL
W0<0kno_0P3emOD30::RXP<0@T920P0A4A4301DE5@80=CDe0`1GEeL306][J`80KFe]0`1PH60204a<
C0<0=CDe0`0P8202010@40<01@D50P010@4300@410<03Ph>0P0O7al303Lg=`<0E5AD0P1bLW8308b<
S080VIVI0`2CTi<307AdM080@d=30`0E5AD200X:2P<0>c/k0`2JVYX20>CTi0<0emOG0P2CTi<3092@
T0<0_[jn0P2;Rh/304=3@`<0>3Ph0P1@D50309:BTP80bLW90`2YZJT304Q8B080BDU90`2g]kL207in
OP<0KVi^0`26QXH207emOBX0OWin1@3oool100000040oooo000?0?ooo`8000000`3ooolZ07inOP80
Ogmo0`20P80307AdM080RHV90`2HV9P205QHF0<0De=C0`2SXj<20<W9b@<0UiNG0P1;Bd/303De=@<0
:RXZ0P1mOGd30>7Qh@<0TI6A0P2LW9`30>o_k`80mOGe0`2RXZ830492@P804A4A0`0E5AD203De=@<0
EeMG0`1[Jf/206e]K@<0H61P0P1<C4`303De=@<0820P0P0@410300D51@800@410`0410@300h>3P80
7alO0`0g=cL305ADE080LW9b0`2<S8`209VIV@<0Ti>C0`1dM7@204=3@`<05ADE0P0:2PX303/k>`<0
VYZJ0P3Ti>@30=OGe`80Ti>C0`2@T9030;jn_P80Rh^;0`13@d<303Ph>080D51@0`2BTY820<W9b@<0
ZJVY0`18B4P204U9B@<0]kNg0P1nOWh306i^KP<0QXJ60P1mOGdZ07inOP@0oooo0P0000010?ooo`00
3`3oool00`000000oooo0?ooo`020?ooobX0OWin0P20P80307]kN`<0N7Qh0P2JVYX307YjNP80B4Q8
0`1nOWh30<?3``80/;2`0`1TI6@2030`<0<0@4100`17AdL20=cLg0<0aLG50`26QXH20<C4a0<0k^k^
0P2b/[8304Y:BP<01PH60P000003028R8P80C4a<0`1XJ6P306i^KP80I6AT0`1?Cdl203Lg=`<0820P
0`0=3@d=000000801PH60`0I6AT303<c<`80DU9B0`1cLg<208n?S`<0Vi^K0`2@T90206YZJP<0<S8b
0P0410@300H61P<0BTY:0P2/[:`30=cLg080/K6a0`25QHD30;6a/@80_[jn0`1]KFd30410@080>c/k
0`1RHV820:b/[0<0b<S80`20P80203<c<`<0PH610P2b/[8305aLG0<0R8R80P1mOGdZ07inOPD0oooo
0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooobX0OWin0P20P80307]kN`<0N7Qh0P2J
VYX307YjNP80B4Q80`1nOWh30<?3``80/;2`0`1TI6@2030`<0<0@4100`17AdL20=cLg0<0aLG50`26
QXH20<C4a0<0k^k^0P2b/[8304Y:BP<01PH60P000003028R8P80C4a<0`1XJ6P306i^KP80I6AT0`1?
Cdl203Lg=`<0820P0`0=3@d=000000801PH60`0I6AT303<c<`80DU9B0`1cLg<208n?S`<0Vi^K0`2@
T90206YZJP<0<S8b0P0410@300H61P<0BTY:0P2/[:`30=cLg080/K6a0`25QHD30;6a/@80_[jn0`1]
KFd30410@080>c/k0`1RHV820:b/[0<0b<S80`20P80203<c<`<0PH610P2b/[8305aLG0<0R8R80P1m
OGdZ07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooobX0OWin0P20P803
07]kN`<0N7Qh0P2JVYX307YjNP80B4Q80`1nOWh30<?3``80/;2`0`1TI6@2030`<0<0@4100`17AdL2
0=cLg0<0aLG50`26QXH20<C4a0<0k^k^0P2b/[8304Y:BP<01PH60P000003028R8P80C4a<0`1XJ6P3
06i^KP80I6AT0`1?Cdl203Lg=`<0820P0`0=3@d=000000801PH60`0I6AT303<c<`80DU9B0`1cLg<2
08n?S`<0Vi^K0`2@T90206YZJP<0<S8b0P0410@300H61P<0BTY:0P2/[:`30=cLg080/K6a0`25QHD3
0;6a/@80_[jn0`1]KFd30410@080>c/k0`1RHV820:b/[0<0b<S80`20P80203<c<`<0PH610P2b/[83
05aLG0<0R8R80P1mOGdZ07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oo
obL0OWin0`1oOgl20820P0<0MGEe0`29RHT209BDU0<0EeMG0P1MGEd30:f][@<0`L710P21PH4303/k
>`80>CTi0`19BDT309nOW`80m_Kf0`2DU9@309NGU`80f=SH0`30`<0205aLG0<01@D50`000002018B
4P<0A4A40P1ZJVX307UiN@<0LW9b0P1OGel304M7A`80<C4a0`0P820301<C4`802`/;0`051@D20082
0P<00@410`030`<200P820<04Q8B0`0S8b<203`l?0<0G5aL0P1mOGd309VIV@<0Xj>S0P2ATI43065Q
H@808R8R0`00000300`<3080H61P0`2i^KT20<G5a@<0Shn?0`2BTY820<c<c0<0YZJV0`1LG5`203hn
?P<0@D510P1lO7`30<71`@<0]KFe0P1BDU8304E5A@80/[:b0`1mOGd3075aL@80Q8B40`1mOGdW07in
OPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooobL0OWin0`1oOgl20820P0<0
MGEe0`29RHT209BDU0<0EeMG0P1MGEd30:f][@<0`L710P21PH4303/k>`80>CTi0`19BDT309nOW`80
m_Kf0`2DU9@309NGU`80f=SH0`30`<0205aLG0<01@D50`000002018B4P<0A4A40P1ZJVX307UiN@<0
LW9b0P1OGel304M7A`80<C4a0`0P820301<C4`802`/;0`051@D200820P<00@410`030`<200P820<0
4Q8B0`0S8b<203`l?0<0G5aL0P1mOGd309VIV@<0Xj>S0P2ATI43065QH@808R8R0`00000300`<3080
H61P0`2i^KT20<G5a@<0Shn?0`2BTY820<c<c0<0YZJV0`1LG5`203hn?P<0@D510P1lO7`30<71`@<0
]KFe0P1BDU8304E5A@80/[:b0`1mOGd3075aL@80Q8B40`1mOGdW07inOPD0oooo0@0000010?ooo`00
3`3oool00`000000oooo0?ooo`020?ooobL0OWin0`20P80207]kN`<0NGUi0`2GUiL207IfMP<0CDe=
0P2:RXX30<?3``<0WYjN0P1@D50302h^;P80EEEE0`1[Jf/30?Cdm080alO70`1kNg/30;2`/080a<C4
0`1cLg<2010@40<000000`030`<203dm?@<0Kfm_0P28R8P308J6QP<0MGEe0P1OGel304];B`80?3`l
0`0d=3@3030`<080;Rh^0`0];Bd202`/;0<0:BTY0`0V9RH202<S8`<08R8R0`0X:2P203De=@<0C4a<
0P1[Jf/308f=S@<0Z:RX0P2^[Zh309:BTP80EeMG0`0C4a<3000000806a/K0`1iNGT20;Zj^P<0XZ:R
0`1nOWh20;:b/P<0clo?0`2:RXX205ADE0<0>CTi0P1@D50309^KV`<0b/[:0P2=SHd303Lg=`80Ng]k
0`2_[jl305mOG`80QhN70`1mOGdW07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?oo
o`020?ooobL0OWin0`20P80207]kN`<0NGUi0`2GUiL207IfMP<0CDe=0P2:RXX30<?3``<0WYjN0P1@
D50302h^;P80EEEE0`1[Jf/30?Cdm080alO70`1kNg/30;2`/080a<C40`1cLg<2010@40<000000`03
0`<203dm?@<0Kfm_0P28R8P308J6QP<0MGEe0P1OGel304];B`80?3`l0`0d=3@3030`<080;Rh^0`0]
;Bd202`/;0<0:BTY0`0V9RH202<S8`<08R8R0`0X:2P203De=@<0C4a<0P1[Jf/308f=S@<0Z:RX0P2^
[Zh309:BTP80EeMG0`0C4a<3000000806a/K0`1iNGT20;Zj^P<0XZ:R0`1nOWh20;:b/P<0clo?0`2:
RXX205ADE0<0>CTi0P1@D50309^KV`<0b/[:0P2=SHd303Lg=`80Ng]k0`2_[jl305mOG`80QhN70`1m
OGdW07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooobL0OWin0`20P802
07]kN`<0NGUi0`2GUiL207IfMP<0CDe=0P2:RXX30<?3``<0WYjN0P1@D50302h^;P80EEEE0`1[Jf/3
0?Cdm080alO70`1kNg/30;2`/080a<C40`1cLg<2010@40<000000`030`<203dm?@<0Kfm_0P28R8P3
08J6QP<0MGEe0P1OGel304];B`80?3`l0`0d=3@3030`<080;Rh^0`0];Bd202`/;0<0:BTY0`0V9RH2
02<S8`<08R8R0`0X:2P203De=@<0C4a<0P1[Jf/308f=S@<0Z:RX0P2^[Zh309:BTP80EeMG0`0C4a<3
000000806a/K0`1iNGT20;Zj^P<0XZ:R0`1nOWh20;:b/P<0clo?0`2:RXX205ADE0<0>CTi0P1@D503
09^KV`<0b/[:0P2=SHd303Lg=`80Ng]k0`2_[jl305mOG`80QhN70`1mOGdW07inOPD0oooo0@000001
0?ooo`003`3oool00`000000oooo0?ooo`020?ooobL0OWin0`1oOgl207MgM`<0R8R80`2@T90205MG
E`<0IVIV0P2c/k<30;Fe]@<0K6a/0P0c<c<304E5A@80Jf][0`2j^[X30?clo080QXJ60`22PX830;Jf
]P80Rh^;0`0U9BD5000000<0=SHf0P1eMGD309RHV080WYjN0`2?Shl307YjNP80IfMW0`1KFe/205MG
E`<0F5QH0`1MGEd206=SH`<0IVIV0P1WIfL306=SH`<0Fe]K0P1ADE4304M7A`<0@4100P0o?cl304Q8
B080GUiN0`1mOGd30:2PX080^[Zj0`2i^KT209:BTP<0C4a<0`071`L2000000<0=3@d0P2>SXh30:^[
Z`<0P8200P28R8P30<o?c`<0^k^k0P1gMgL304Y:BP80>CTi0`1YJFT30;Ng]`80_Kfm0`1MGEd204=3
@`<0[Zj^0`1mOGd207AdM0<0PX:20P1mOGdU07inOPD0oooo0@0000010?ooo`000P3oool2000000@0
oooo0P0000050?ooo`030000003oool0oooo0080oooo9`1nOWh307moO`80MgMg0`28R8P3092@T080
EeMG0`1VIVH20;>c/`<0]KFe0`1/K6`203<c<`<0ADE50P1[Jf/30;Zj^P<0o?cl0P26QXH308:2PP<0
][Jf0P2;Rh/302DU9@D000000`0f=SH207EeM@<0V9RH0P2NWYh308n?S`<0NWYj0P1WIfL305]KF`80
EeMG0`1HF5P305eMG@80Hf=S0`1VIVH206MWI`<0Hf=S0`1KFe/2055AD@<0AdM70`10@40203lo?`<0
B4Q80P1NGUh307emO@<0X:2P0P2j^[X30;Vi^@80TY:B0`1<C4`300L71`8000000`0d=3@208j>SP<0
Zj^[0`20P80208R8R0<0clo?0`2k^k/207MgM`<0BTY:0P0i>CT306UYJ@<0]kNg0P2m_Kd305eMG@80
@d=30`2^[Zh307emO@80M7Ad0`22PX8207emOBD0OWin1@3oool100000040oooo00001@3oool00000
0?ooo`3oool000000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`020?oo
obL0OWin0`1oOgl207MgM`<0R8R80`2@T90205MGE`<0IVIV0P2c/k<30;Fe]@<0K6a/0P0c<c<304E5
A@80Jf][0`2j^[X30?clo080QXJ60`22PX830;Jf]P80Rh^;0`0U9BD5000000<0=SHf0P1eMGD309RH
V080WYjN0`2?Shl307YjNP80IfMW0`1KFe/205MGE`<0F5QH0`1MGEd206=SH`<0IVIV0P1WIfL306=S
H`<0Fe]K0P1ADE4304M7A`<0@4100P0o?cl304Q8B080GUiN0`1mOGd30:2PX080^[Zj0`2i^KT209:B
TP<0C4a<0`071`L2000000<0=3@d0P2>SXh30:^[Z`<0P8200P28R8P30<o?c`<0^k^k0P1gMgL304Y:
BP80>CTi0`1YJFT30;Ng]`80_Kfm0`1MGEd204=3@`<0[Zj^0`1mOGd207AdM0<0PX:20P1mOGdU07in
OPD0oooo0@0000010?ooo`000`3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?oo
o`3oool00P3oool300000080oooo9@1nOWh207moO`<0O7al0P1jNWX309BDU0<0Lg=c0P1BDU8309BD
U080`<300`2;Rh/30492@P80=3@d0`1_Kfl208^;R`<0oooo0`2j^[X206=SH`<0U9BD0`2IVIT204=3
@`D000000`0];Bd307MgM`80Z:RX0`2e]KD20:VYZ@<0Ti>C0`1oOgl207EeM@<0M7Ad0P1jNWX308J6
QP<0TY:B0P2MWId30:BTY080YJFU0`2QXJ4309RHV080RHV90`1hN7P306IVIP80FEUI0`1CDe<205UI
F@<0KVi^0`2?Shl20;Bd]0<0bl_;0P32`/8308n?S`<0?clo0P00000300D51@80DU9B0`2IVIT308f=
S@80K6a/0`2WYjL30=KFeP80XZ:R0`1[Jf/203lo?`<0@T920`29RHT20<O7a`<0VIVI0P0k>c/307Mg
M`<0[:b/0P1RHV8308J6QP80OGem9@1nOWh40?ooo`8000000@3oool000@0oooo00@000000?ooo`3o
ool000000P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo9@1nOWh207mo
O`<0O7al0P1jNWX309BDU0<0Lg=c0P1BDU8309BDU080`<300`2;Rh/30492@P80=3@d0`1_Kfl208^;
R`<0oooo0`2j^[X206=SH`<0U9BD0`2IVIT204=3@`D000000`0];Bd307MgM`80Z:RX0`2e]KD20:VY
Z@<0Ti>C0`1oOgl207EeM@<0M7Ad0P1jNWX308J6QP<0TY:B0P2MWId30:BTY080YJFU0`2QXJ4309RH
V080RHV90`1hN7P306IVIP80FEUI0`1CDe<205UIF@<0KVi^0`2?Shl20;Bd]0<0bl_;0P32`/8308n?
S`<0?clo0P00000300D51@80DU9B0`2IVIT308f=S@80K6a/0`2WYjL30=KFeP80XZ:R0`1[Jf/203lo
?`<0@T920`29RHT20<O7a`<0VIVI0P0k>c/307MgM`<0[:b/0P1RHV8308J6QP80OGem9@1nOWh50?oo
o`4000000@3oool000050?ooo`000000oooo0?ooo`0000000P3oool010000000oooo0?ooo`000004
0?ooo`030000003oool0oooo0080oooo9@1nOWh207moO`<0N7Qh0P27QhL308f=S@<0F5QH0P1_Kfl3
0;Ng]`80Z:RX0`1JFUX3038b<P80F5QH0`2:RXX20<_;b`<0m?Cd0`1[Jf/206MWI`<0UYJF0`1SHf<2
00T92@<000000P0S8b<307IfMP<0];Bd0P3;bl/30<;2`P80ZZZZ0`2CTi<508N7Q`<0T92@0P2PX:03
0;:b/P<0`/;20P3?cll30=SHf080fm_K0`3GemL30<k>cP80_[jn0`2ZZZX309>CT`80O7al0`1YJFT2
05mOG`<0IFEU0`1kNg/20:2PX0<0alO70P3Kfm/30<S8b0<0QhN70P0a<C43000000807AdM0`1^KVh3
09:BTP80Jf][0`1dM7@30<O7a`80b<S80`2@T90205aLG0<0=cLg0`1GEeL20:ZZZP<0`l?30P1WIfL3
0492@P<0ZZZZ0P1lO7`307MgM`80PH619@1nOWh50?ooo`4000000@3oool00080oooo0P0000040?oo
o`8000001@3oool00`000000oooo0?ooo`020?ooobD0OWin0P1oOgl307QhN080QhN70`2=SHd305QH
F080Kfm_0`2g]kL20:RXZ0<0FUYJ0`0b<S8205QHF0<0RXZ:0P3;bl/30?Cdm0<0Jf][0P1WIfL309JF
UP<0Hf=S0P092@T3000000808b<S0`1fMWH30;Bd]080bl_;0`32`/820:ZZZP<0Ti>C1@27QhL3092@
T080X:2P0`2b/[830<;2`P80clo?0`3Hf=P20=_Kf`<0emOG0`3>c/h20;jn_P<0ZZZZ0`2CTi<207al
O0<0JFUY0P1OGel306EUI@<0Ng]k0P2PX:030<O7a`80fm_K0`38b<P308N7Q`80<C4a0`00000201dM
7@<0KVi^0`2BTY8206][J`<0M7Ad0`37alL20<S8b0<0T92@0P1LG5`303Lg=`<0EeMG0P2ZZZX30<?3
``80IfMW0`12@T830:ZZZP80O7al0`1gMgL20861PBD0OWin1@3oool100000040oooo000?0?ooo`03
0000003oool0oooo0080oooo9@1nOWh207moO`<0N7Qh0P27QhL308f=S@<0F5QH0P1_Kfl30;Ng]`80
Z:RX0`1JFUX3038b<P80F5QH0`2:RXX20<_;b`<0m?Cd0`1[Jf/206MWI`<0UYJF0`1SHf<200T92@<0
00000P0S8b<307IfMP<0];Bd0P3;bl/30<;2`P80ZZZZ0`2CTi<508N7Q`<0T92@0P2PX:030;:b/P<0
`/;20P3?cll30=SHf080fm_K0`3GemL30<k>cP80_[jn0`2ZZZX309>CT`80O7al0`1YJFT205mOG`<0
IFEU0`1kNg/20:2PX0<0alO70P3Kfm/30<S8b0<0QhN70P0a<C43000000807AdM0`1^KVh309:BTP80
Jf][0`1dM7@30<O7a`80b<S80`2@T90205aLG0<0=cLg0`1GEeL20:ZZZP<0`l?30P1WIfL30492@P<0
ZZZZ0P1lO7`307MgM`80PH619@1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool0
0P3ooolR07inOP<0Ogmo0P1lO7`307]kN`80TY:B0`1bLW8305QHF080Vi^K0`2i^KT207YjNP<0>SXj
0`12@T8208V9R@<0XZ:R0P3lo?`30:6QX@<0ADE50P1jNWX307]kN`<0:BTY0P00000301TI6@80Kfm_
0`2k^k/30=kNgP80f=SH0`2m_Kd20:6QX@<0TI6A0`2?Shl209VIV@<0ZZZZ0P2o_kl30=;BdP<0hn?S
0P3`l?030?Shn080no_k0`3jn_X30?;blP80iNGU0`3Bd]830;^k^`80X:2P0`25QHD206m_K`<0IFEU
0`1[Jf/208J6QP<0[jn_0P3IfMT30>SXj0<0bLW90P1jNWX302HV9P802@T90`0m?Cd30820P080NGUi
0`1FEUH309>CT`80dm?C0`2d];@208>3P`<0Bd];0`0j>SX207MgM`<0`/;20P2SXj<30410@0<0M7Ad
0P2XZ:P306EUI@80QHF50`1mOGdR07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?oo
o`020?ooob80OWin0`1oOgl207alO0<0Ng]k0P2BTY83079bLP<0F5QH0P2KVi/30;Vi^@80NWYj0`0j
>SX30492@P80RHV90`2RXZ820?clo0<0XJ6Q0`15ADD207YjNP<0Ng]k0`0Y:BT2000000<06ATI0P1_
Kfl30;^k^`<0g]kN0P3Hf=P30;fm_@80XJ6Q0`2ATI4308n?S`80VIVI0`2ZZZX20;no_`<0d];B0`3S
hn<20?3`l0<0n?Sh0P3kno/30?[jnP<0l_;b0P3UiND30=;BdP<0^k^k0P2PX:0308F5Q@80Kfm_0`1U
IFD306][J`80QXJ60`2_[jl20=WIf@<0j>SX0`39bLT207YjNP<09RHV0P092@T303dm?@<0P8200P1i
NGT305IFEP<0Ti>C0P3Cdm<30;Bd]080Ph>30`1;Bd/303Xj>P80MgMg0`32`/820:>SX`<0@4100`1d
M7@20:RXZ0<0IFEU0P25QHD307emOB80OWin1@3oool100000040oooo000?0?ooo`030000003oool0
oooo0080oooo8P1nOWh307moO`80NGUi0`26QXH208Z:RP<0FUYJ0`1gMgL20;Ng]`<0VYZJ0P1=CDd3
03Lg=`<0KVi^0P2TY:@30=3@d080h>3P0`1:BTX304i>CP80P8200`1>CTh300T92@804Q8B0`1SHf<2
0;^k^`H0jn_[0P3=cLd30:^[Z`80U9BD0`2>SXh309JFUP80YjNW0`2l_;`20=3@d0<0hN7Q0`3_knl2
0?Shn0<0o_kn1@3oool30?gmo@80mOGe0`3Xj>P30=GEe@80_Kfm0`2QXJ4208B4Q0<0KFe]0`1SHf<2
06e]K@<0SXj>0P2n_[h30>SXj0<0lO7a0P32`/8306YZJP807alO0`0M7Ad305eMG@80PH610`1IFET3
05aLG080]KFe0`3;bl/20:FUY@<0LG5a0`0l?3`204Y:BP<0W9bL0P35aLD3071`L0<0@T920P2WYjL3
07]kN`80NGUi0`20P80R07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oo
ob80OWin0`1oOgl207UiN@<0QXJ60P2:RXX305YJFP<0MgMg0P2g]kL309ZJVP80CDe=0`0g=cL306i^
KP80Y:BT0`3@d=020>3Ph0<0BTY:0`1>CTh20820P0<0CTi>0`092@T2018B4P<0Hf=S0P2k^k/60>_[
j`80cLg=0`2[Zj/209BDU0<0SXj>0`2FUYH20:NWY`<0_;bl0P3@d=030>7Qh@<0kno_0P3hn?P30?kn
oPD0oooo0`3moOd20?Gem@<0j>SX0`3EeMD20;fm_@<0XJ6Q0P24Q8@306e]K@<0Hf=S0P1]KFd308j>
SP80_[jn0`3Xj>P30?7al@80`/;20`1ZJVX201lO7`<07AdM0`1MGEd20861P@<0FEUI0`1LG5`20;Fe
]@<0bl_;0P2UYJD3075aL@<0?3`l0P1:BTX309bLW080aLG50`1`L7030492@P80YjNW0`1kNg/207Ui
N@<0P8208P1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooolR07inOP<0
Ogmo0P1iNGT308J6QP80RXZ:0`1JFUX307MgM`80]kNg0`2JVYX204e=C@<0=cLg0`1^KVh20:BTY0<0
d=3@0P3Ph>0304Y:BP<0CTi>0P20P80304i>CP<02@T90P0B4Q8306=SH`80^k^k1P3[jn/20<g=c@<0
Zj^[0P2DU9@308j>SP<0UYJF0P2WYjL30;bl_080d=3@0`3QhN430>o_k`80n?Sh0`3no_h50?ooo`<0
oOgm0P3emOD30>SXj0<0eMGE0P2m_Kd30:6QX@80Q8B40`1]KFd306=SH`80KFe]0`2>SXh20;jn_P<0
j>SX0`3alO420<;2`P<0JVYZ0P0O7al301dM7@<0GEeM0P21PH4305UIF@<0G5aL0P2e]KD30<_;b`80
YJFU0`1aLG4303`l?080BTY:0`2LW9`20<G5a@<0L71`0`12@T820:NWY`<0Ng]k0P1iNGT30820P280
OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo7`1nOWh307moO`<0OGem
0P1kNg/308n?S`80LG5a0`1MGEd30:2PX080/[:b0`1[Jf/203Ph>0<0De=C0`2QXJ420;2`/0<0k^k^
0P21PH4302PX:0<0JVYZ0P1]KFd302PX:0<04a<C0P1DE5@30;Bd]080lo?c0`3jn_X30=_Kf`80/K6a
0`2BTY8208J6QP<0RXZ:0`2JVYX20:f][@<0`<300P3?cll30=[JfP<0h^;R0P3Vi^H30>WYj@80jn_[
1P3]kNd20>c/k0<0inOW0`3MgMd20<c<c0<0][Jf0P2IVIT307alO0<0IFEU0P1MGEd306a/K080UIFE
0`3;bl/30?Gem@80lo?c0`2e]KD205QHF0<08B4Q0`0k>c/207EeM@<0KFe]0`12@T8207YjNP<0a/K6
0P2m_Kd309VIV@<0Fe]K0P0g=cL306MWI`80^k^k0`2[Zj/304E5A@80LW9b0`2UYJD206QXJ0<0QHF5
0`1mOGdO07inOPD0oooo0@0000010?ooo`003`3oool2000000<0oooo7`1nOWh307moO`<0OGem0P1k
Ng/308n?S`80LG5a0`1MGEd30:2PX080/[:b0`1[Jf/203Ph>0<0De=C0`2QXJ420;2`/0<0k^k^0P21
PH4302PX:0<0JVYZ0P1]KFd302PX:0<04a<C0P1DE5@30;Bd]080lo?c0`3jn_X30=_Kf`80/K6a0`2B
TY8208J6QP<0RXZ:0`2JVYX20:f][@<0`<300P3?cll30=[JfP<0h^;R0P3Vi^H30>WYj@80jn_[1P3]
kNd20>c/k0<0inOW0`3MgMd20<c<c0<0][Jf0P2IVIT307alO0<0IFEU0P1MGEd306a/K080UIFE0`3;
bl/30?Gem@80lo?c0`2e]KD205QHF0<08B4Q0`0k>c/207EeM@<0KFe]0`12@T8207YjNP<0a/K60P2m
_Kd309VIV@<0Fe]K0P0g=cL306MWI`80^k^k0`2[Zj/304E5A@80LW9b0`2UYJD206QXJ0<0QHF50`1m
OGdO07inOP@0oooo0P0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooal0OWin0`1oOgl3
07emO@80Ng]k0`2?Shl2075aL@<0GEeM0`2PX:020;:b/P<0Jf][0P0h>3P305=CD`<0XJ6Q0P2`/;03
0>k^kP80PH610`0X:2P306YZJP80KFe]0`0X:2P301<C4`80E5AD0`2d];@20??cl`<0n_[j0`3Kfm/2
0;6a/@<0TY:B0P26QXH308Z:RP<0VYZJ0P2][Jd30<30`080clo?0`3Jf]X30>;RhP80i^KV0`3YjNT2
0>_[j`H0kNg]0P3/k>`30>OWi`<0gMgM0P3<c<`30;Jf]P80VIVI0`1lO7`306EUI@80GEeM0`1/K6`2
09FEU@<0bl_;0`3emOD20??cl`<0]KFe0P1HF5P3024Q8@<0>c/k0P1eMGD306e]K@<0@T920P1jNWX3
0<K6aP80_Kfm0`2IVIT305]KF`80=cLg0`1WIfL20;^k^`<0Zj^[0`15ADD2079bLP<0YJFU0P1XJ6P3
08F5Q@<0OGem7`1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooolO07in
OP<0Ogmo0`1jNWX208J6QP<0QhN70P1LG5`307inOP<0][Jf0P2=SHd304A4A080@T920`25QHD30;Ng
]`80bLW90`34a<@202XZ:P<0?Cdm0`1hN7P204e=C@<07alO0`15ADD20:FUY@<0l_;b0P3oool30>SX
j0<0][Jf0P2=SHd507UiN@<0QhN70`2IVIT20:ZZZP<0][Jf0P2m_Kd60<30`0L0_[jn0`31`L430<K6
aP80b/[:0`3>c/h30<o?c`80b/[:0`2m_Kd20:RXZ0<0SHf=0`1_Kfl205QHF0<0E5AD0P1ZJVX309bL
W0<0f=SH0P3no_h30>k^kP80XJ6Q0`19BDT302h^;P80G5aL0`1kNg/304e=C@80A4A40`2NWYh20<G5
a@<0];Bd0`26QXH204I6AP<0@4100P2=SHd30<K6aP<0NGUi0P13@d<30:BTY080NWYj0`1kNg/30820
P1l0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo7`1nOWh307moO`<0
NWYj0P26QXH308N7Q`80G5aL0`1nOWh30;Jf]P80SHf=0`14A4@20492@P<0QHF50`2g]kL20<W9b@<0
a<C40P0Z:RX303dm?@<0N7Qh0P1=CDd301lO7`<0ADE50P2UYJD30?;blP80oooo0`3Xj>P30;Jf]P80
SHf=1@1iNGT308N7Q`<0VIVI0P2ZZZX30;Jf]P80_Kfm1P30`<070;jn_P<0`L710`36a/H20<[:bP<0
c/k>0`3?cll20<[:bP<0_Kfm0P2XZ:P308f=S@<0Kfm_0P1HF5P305ADE080JVYZ0`2LW9`30=SHf080
o_kn0`3^k^h20:6QX@<0BDU90`0^;Rh205aLG0<0Ng]k0`1=CDd204A4A0<0WYjN0P35aLD30;Bd]0<0
QXJ60P16ATH30410@080SHf=0`36a/H307UiN@80@d=30`2TY:@207YjNP<0Ng]k0`20P80O07inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooad0OWin0P1oOgl307emO@<0O7al
0P2=SHd3071`L080Hf=S0`2TY:@30:VYZ@80GUiN0`0j>SX206MWI`H0];Bd0P3Fe]H3061PH0804a<C
0`1VIVH3071`L080>3Ph0`0m?Cd3092@T080jNWY0`3oool20??cl`<0^k^k0`27QhL206YZJP<0IFEU
0P1`L70308:2PP<0TY:B0P2LW9`309nOW`80WIfM0`2GUiL308n?S`80RHV91@25QHD308R8R0<0Shn?
0P2IVIT30:FUY@<0[jn_1@2f][H20:f][@<0VYZJ0`1nOWh205mOG`<0BTY:0P19BDT306QXJ0<0Y:BT
0P3Ti>@30?ooo`80h>3P0`2:RXX304=3@`80AdM70`1gMgL306YZJP80<c<c0`1PH6020;>c/`<0_kno
0`2[Zj/206e]K@<0>CTi0P1IFET30;:b/P<0/K6a0P1:BTX3071`L080XZ:R0`1ZJVX308B4Q080OGem
7@1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooolM07inOP80Ogmo0`1m
OGd307alO080SHf=0`1`L70206=SH`<0Y:BT0`2YZJT205iNGP<0>SXj0P1WIfL60;Bd]080e]KF0`1P
H60201<C4`<0IVIV0`1`L70203Ph>0<0?Cdm0`2@T9020>WYj@<0oooo0P3clo<30;^k^`<0QhN70P1Z
JVX306EUI@80L71`0`22PX8309:BTP80W9bL0`2OWil209fMW@<0UiNG0`2?Shl208V9R@D0QHF50`28
R8P308n?S`80VIVI0`2UYJD30:n_[`D0][Jf0P2][Jd309ZJVP<0OWin0P1OGel304Y:BP80BDU90`1X
J6P30:BTY080i>CT0`3oool20>3Ph0<0RXZ:0`13@d<204M7A`<0MgMg0`1ZJVX203<c<`<0H61P0P2c
/k<30;no_`<0Zj^[0P1]KFd303Ti>@80FEUI0`2b/[830;6a/@80BTY:0`1`L7020::RXP<0JVYZ0`24
Q8@207emOAd0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo7@1nOWh2
07moO`<0OGem0`1lO7`208f=S@<0L71`0P1SHf<30:BTY0<0ZJVY0P1NGUh303Xj>P80IfMW1P2d];@2
0=KFeP<0H61P0P0C4a<306IVIP<0L71`0P0h>3P303dm?@<0T92@0P3YjNT30?ooo`80lo?c0`2k^k/3
08N7Q`80JVYZ0`1UIFD2071`L0<0PX:20`2BTY8209bLW0<0WinO0P2MWId309NGU`<0Shn?0P29RHT5
08F5Q@<0R8R80`2?Shl209VIV@<0YJFU0`2_[jl50;Jf]P80[Jf]0`2JVYX307inOP80GemO0`1:BTX2
04U9B@<0J6QX0`2TY:@20>CTi0<0oooo0P3Ph>0308Z:RP<0@d=30P17AdL307MgM`<0JVYZ0P0c<c<3
061PH080/k>c0`2o_kl30:^[Z`80KFe]0`0i>CT205UIF@<0/[:b0`2a/K4204Y:BP<0L71`0P2RXZ83
06YZJP<0Q8B40P1mOGdM07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oo
oad0OWin0P1oOgl307YjNP<0QHF50P26QXH305mOG`80Ph>30`2c/k<30820P080@4100`1@D50209VI
V@<0`L710`2i^KT20:BTY0<0410@0P0i>CT30820P0<0Fe]K0P10@40307YjNP<0f=SH0P3oool30?cl
o080`L710`23Ph<305]KF`80D51@0`1JFUX206a/K0<0O7al0`25QHD208J6QP<0P8200P1eMGD306QX
J0<0G5aL0P1BDU8304a<C080BTY:0`1>CTh305IFEP80Hf=S0`1cLg<308B4Q080UIFE0`2QXJ420:JV
YP<0WinO0`2<S8`206m_K`<0Cdm?0P0j>SX30410@0<0JFUY0P2][Jd30>k^kP80oooo0`3<c<`307Ad
M080B4Q80`1VIVH30861P@80BTY:0`0a<C4208B4Q0<0]kNg0`2l_;`209ZJVP<0De=C0P0j>SX307mo
O`<0a<C40P21PH4304A4A080XZ:R0`1iNGT307alO080Ogmo7@1nOWh50?ooo`4000000@3oool000l0
oooo00<000000?ooo`3oool00P3ooolM07inOP80Ogmo0`1jNWX308F5Q@80QXJ60`1OGel208>3P`<0
/k>c0`20P8020410@0<0D51@0P2IVIT30<71`@<0^KVi0P2TY:@3010@4080>CTi0`20P80305]KF`80
@4100`1jNWX30=SHf080oooo0`3lo?`20<71`@<0Ph>30`1KFe/2051@D0<0FUYJ0P1/K6`307alO0<0
QHF50P26QXH30820P080MGEe0`1XJ6P305aLG080DU9B0`1<C4`204Y:BP<0CTi>0`1FEUH206=SH`<0
Lg=c0`24Q8@209FEU@<0XJ6Q0P2VYZH309nOW`<0S8b<0P1_Kfl304m?C`80>SXj0`10@40306UYJ@80
[Jf]0`3^k^h20?ooo`<0c<c<0`1dM7@204Q8B0<0IVIV0`21PH4204Y:BP<0<C4a0P24Q8@30;Ng]`<0
_;bl0P2JVYX305=CD`80>SXj0`1oOgl30<C4a080PH610`14A4@20::RXP<0NGUi0`1lO7`207moOad0
OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo7@1nOWh207moO`<0NWYj
0`25QHD208J6QP<0GemO0P23Ph<30;>c/`<0P8200P10@403051@D080VIVI0`31`L430;Vi^@80Y:BT
0`0@410203Ti>@<0P8200`1KFe/20410@0<0NWYj0`3Hf=P20?ooo`<0o?cl0P31`L4308>3P`<0Fe]K
0P1@D50305YJFP80K6a/0`1lO7`308F5Q@80QXJ60`20P80207EeM@<0J6QX0`1LG5`2059BDP<0C4a<
0P1:BTX304i>CP<0EUIF0P1SHf<307=cL`<0Q8B40P2EUID30:6QX@80YZJV0`2OWil308b<S080Kfm_
0`1?Cdl203Xj>P<0@4100`1YJFT20:f][@<0k^k^0P3oool30<c<c0<0M7Ad0P18B4P306IVIP<0PH61
0P1:BTX3034a<@80Q8B40`2g]kL30;bl_080VYZJ0`1CDe<203Xj>P<0Ogmo0`34a<@20861P@<0A4A4
0P2RXZ8307UiN@<0O7al0P1oOglM07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?oo
o`020?oooaX0OWin0`1oOgl207emO@<0O7al0`2<S8`2071`L0<0IfMW0P2UYJD309nOW`<0EEEE0P11
@D4307]kN`80`L710`2_[jl30;Vi^@80@d=30`092@T2075aL@<0Ogmo0`1ADE4206QXJ0<0_kno1@3o
ool30<W9b@80PH610`1?Cdl303dm?@80ADE50`1HF5P206][J`<0MGEe0`1fMWH206i^KP<0H61P0P1?
Cdl303lo?`<0<30`0P0U9BD301hN7P8071`L0`0N7Qh302HV9P80<S8b0`13@d<305QHF080KVi^0`24
Q8@209BDU0<0Vi^K0`2EUID20861P@<0HF5Q0P0o?cl302/[:`<0>3Ph0P1/K6`30;Vi^@D0mOGe0`2b
/[8206EUI@<0FUYJ0`23Ph<207=cL`<0;Bd]0P1:BTX309^KV`D0^;Rh0`20P80203lo?`<0CTi>0`2X
Z:P20;Jf]P<0Cdm?0P1`L70309jNWP<0KFe]0P23Ph<M07inOPD0oooo0@0000010?ooo`003`3oool2
000000<0oooo6P1nOWh307moO`80OGem0`1lO7`308b<S080L71`0`1WIfL20:FUY@<0WinO0`1EEED2
0451@@<0Ng]k0P31`L430:n_[`<0^KVi0P13@d<300T92@80LG5a0`1oOgl3055AD@80J6QX0`2o_kl5
0?ooo`<0bLW90P21PH4304m?C`<0?Cdm0P15ADD305QHF080Jf][0`1eMGD307IfMP80KVi^0`1PH602
04m?C`<0?clo0`0`<30202DU9@<07QhN0P0L71`301hN7P<09RHV0P0b<S8304=3@`<0F5QH0P1^KVh3
08B4Q080U9BD0`2KVi/309FEU@80PH610`1QHF4203lo?`<0:b/[0`0h>3P206a/K0<0^KVi1@3emOD3
0;:b/P80IFEU0`1JFUX308>3P`80Lg=c0`0];Bd204Y:BP<0Vi^K1@2h^;P30820P080?clo0`1>CTh3
0:RXZ080][Jf0`1?Cdl2071`L0<0WYjN0`1]KFd208>3Pad0OWin103oool200000040oooo000?0?oo
o`030000003oool0oooo0080oooo6P1nOWh307moO`80Ng]k1P24Q8@2065QH@<0QhN70P2_[jl307Ee
M@<0?clo0P1PH6030:ZZZP80`l?30`2TY:@308J6QP8000000`12@T8209BDU0<0LG5a0`1QHF420:>S
X`<0lo?c0`3oool20=;BdP<0Ph>30P15ADD302/[:`<0<C4a0P17AdL305iNGP80K6a/0`1^KVh306IV
IP80EeMG0`15ADD203<c<`<08b<S0`0E5AD200/;2`<01@D50P030`<300@410<02@T90P0C4a<3024Q
8@<0=3@d0P1;Bd/306ATI080O7al0`2>SXh309NGU`80T92@0`1hN7P205=CD`<0;bl_0`0O7al203@d
=0<0M7Ad0P35aLD30?Ogm`<0h^;R0P2FUYH3069RHP<0MgMg0P2?Shl3059BDP809RHV0`1[Jf/30:FU
Y@80_;bl0`2ZZZX2069RHP<0>3Ph0`1cLg<20<71`@<0QhN70P16ATH309nOW`<0N7Qh0P1nOWh307mo
OaX0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo6P1nOWh307moO`80
Ng]k1P24Q8@2065QH@<0QhN70P2_[jl307EeM@<0?clo0P1PH6030:ZZZP80`l?30`2TY:@308J6QP80
00000`12@T8209BDU0<0LG5a0`1QHF420:>SX`<0lo?c0`3oool20=;BdP<0Ph>30P15ADD302/[:`<0
<C4a0P17AdL305iNGP80K6a/0`1^KVh306IVIP80EeMG0`15ADD203<c<`<08b<S0`0E5AD200/;2`<0
1@D50P030`<300@410<02@T90P0C4a<3024Q8@<0=3@d0P1;Bd/306ATI080O7al0`2>SXh309NGU`80
T92@0`1hN7P205=CD`<0;bl_0`0O7al203@d=0<0M7Ad0P35aLD30?Ogm`<0h^;R0P2FUYH3069RHP<0
MgMg0P2?Shl3059BDP809RHV0`1[Jf/30:FUY@80_;bl0`2ZZZX2069RHP<0>3Ph0`1cLg<20<71`@<0
QhN70P16ATH309nOW`<0N7Qh0P1nOWh307moOaX0OWin1@3oool100000040oooo000?0?ooo`030000
003oool0oooo0080oooo6P1nOWh307moO`80Ng]k1P24Q8@2065QH@<0QhN70P2_[jl307EeM@<0?clo
0P1PH6030:ZZZP80`l?30`2TY:@308J6QP8000000`12@T8209BDU0<0LG5a0`1QHF420:>SX`<0lo?c
0`3oool20=;BdP<0Ph>30P15ADD302/[:`<0<C4a0P17AdL305iNGP80K6a/0`1^KVh306IVIP80EeMG
0`15ADD203<c<`<08b<S0`0E5AD200/;2`<01@D50P030`<300@410<02@T90P0C4a<3024Q8@<0=3@d
0P1;Bd/306ATI080O7al0`2>SXh309NGU`80T92@0`1hN7P205=CD`<0;bl_0`0O7al203@d=0<0M7Ad
0P35aLD30?Ogm`<0h^;R0P2FUYH3069RHP<0MgMg0P2?Shl3059BDP809RHV0`1[Jf/30:FUY@80_;bl
0`2ZZZX2069RHP<0>3Ph0`1cLg<20<71`@<0QhN70P16ATH309nOW`<0N7Qh0P1nOWh307moOaX0OWin
1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo5`1nOWh307moO`<0OGem0P1l
O7`308Z:RP<0L71`0P1/K6`30:JVYP80UYJF0`1>CTh304Y:BP80SXj>0`37alL20:BTY0<0W9bL0`0^
;Rh200`<30<0QhN70P2FUYH306][J`<0RXZ:0P3IfMT30?ooo`<0g=cL0P2:RXX30451@@807AdM0`0P
820303Ti>@80EEEE0`1YJFT206m_K`<0J6QX0`1JFUX204U9B@<0=cLg0P0W9bL301XJ6P<04A4A0P0:
2PX300H61PD010@40`061PH200/;2`<04a<C0`0Q8B4203<c<`<0BDU90P1SHf<307alO0<0T92@0P2G
UiL308j>SP80LG5a0`18B4P3024Q8@805ADE0`0e=CD20820P0<0d=3@0`3alO420<O7a`<0P8200`1]
KFd209>CT`<0Ph>30P0a<C4303Ti>@<0Ph>30P2/[:`30;jn_P80TI6A0`18B4P304E5A@80WYjN0`2i
^KT205ADE0<0Kfm_0`2KVi/206m_K`<0PX:26P1nOWh50?ooo`4000000@3oool000l0oooo00<00000
0?ooo`3oool00P3ooolG07inOP<0Ogmo0`1mOGd207alO0<0RXZ:0`1`L70206a/K0<0YZJV0P2FUYH3
04i>CP<0BTY:0P2>SXh30<O7a`80Y:BT0`2LW9`302h^;P8030`<0`27QhL209JFUP<0Jf][0`2:RXX2
0=WIf@<0oooo0`3Lg=`208Z:RP<0@D510P0M7Ad3020P80<0>CTi0P1EEED306UYJ@80Kfm_0`1XJ6P3
05YJFP80BDU90`0g=cL202LW9`<06QXJ0`0A4A4200X:2P<01PH61@0410@300H61P802`/;0`0C4a<3
024Q8@80<c<c0`19BDT206=SH`<0O7al0`2@T90209NGU`<0SXj>0P1aLG4304Q8B0<08B4Q0P0E5AD3
03De=@80P8200`3@d=030?7al@80alO70`20P80306e]K@80Ti>C0`23Ph<2034a<@<0>CTi0`23Ph<2
0:b/[0<0_[jn0P2ATI4304Q8B0<0ADE50P2NWYh30;Vi^@80E5AD0`1_Kfl309^KV`80Kfm_0`22PX8J
07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooaL0OWin0`1oOgl307]k
N`D0Ph>30`1TI6@208^;R`<0ZZZZ0P1/K6`30451@@<0L71`0P2f][H30;jn_P80Rh^;0`1]KFd30000
0080EUIF0`2^[Zh208N7Q`<0Ng]k0`2k^k/20??cl`<0hn?S0`2EUID20492@P<04a<C0P0A4A4302d]
;@<0D51@0P1ZJVX307EeM@80LW9b0`1VIVH305EEE@80ADE50`0h>3P202h^;P<0:2PX0`0T92@2028R
8P<08B4Q0P0O7al301hN7PD071`L0`0M7Ad3028R8P80:b/[0`0j>SX2051@D0<0JFUY0`23Ph<209JF
UP<0WIfM0P2?Shl306a/K0<0?3`l0P0D51@300l?3`80?3`l0`2?Shl30=SHf080hn?S0`2ZZZX307Ee
M@80QhN70`2SXj<2069RHP<09BDU0`1GEeL208n?S`<0]kNg0P2f][H3079bLP<0>SXj0P1WIfL30;fm
_@80S8b<0`18B4P309jNWP80MgMg0`1oOglJ07inOPD0oooo0@0000010?ooo`003`3oool00`000000
oooo0?ooo`020?oooaL0OWin0`1oOgl307]kN`D0Ph>30`1TI6@208^;R`<0ZZZZ0P1/K6`30451@@<0
L71`0P2f][H30;jn_P80Rh^;0`1]KFd300000080EUIF0`2^[Zh208N7Q`<0Ng]k0`2k^k/20??cl`<0
hn?S0`2EUID20492@P<04a<C0P0A4A4302d];@<0D51@0P1ZJVX307EeM@80LW9b0`1VIVH305EEE@80
ADE50`0h>3P202h^;P<0:2PX0`0T92@2028R8P<08B4Q0P0O7al301hN7PD071`L0`0M7Ad3028R8P80
:b/[0`0j>SX2051@D0<0JFUY0`23Ph<209JFUP<0WIfM0P2?Shl306a/K0<0?3`l0P0D51@300l?3`80
?3`l0`2?Shl30=SHf080hn?S0`2ZZZX307EeM@80QhN70`2SXj<2069RHP<09BDU0`1GEeL208n?S`<0
]kNg0P2f][H3079bLP<0>SXj0P1WIfL30;fm_@80S8b<0`18B4P309jNWP80MgMg0`1oOglJ07inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooaL0OWin0`1oOgl307]kN`D0Ph>3
0`1TI6@208^;R`<0ZZZZ0P1/K6`30451@@<0L71`0P2f][H30;jn_P80Rh^;0`1]KFd300000080EUIF
0`2^[Zh208N7Q`<0Ng]k0`2k^k/20??cl`<0hn?S0`2EUID20492@P<04a<C0P0A4A4302d];@<0D51@
0P1ZJVX307EeM@80LW9b0`1VIVH305EEE@80ADE50`0h>3P202h^;P<0:2PX0`0T92@2028R8P<08B4Q
0P0O7al301hN7PD071`L0`0M7Ad3028R8P80:b/[0`0j>SX2051@D0<0JFUY0`23Ph<209JFUP<0WIfM
0P2?Shl306a/K0<0?3`l0P0D51@300l?3`80?3`l0`2?Shl30=SHf080hn?S0`2ZZZX307EeM@80QhN7
0`2SXj<2069RHP<09BDU0`1GEeL208n?S`<0]kNg0P2f][H3079bLP<0>SXj0P1WIfL30;fm_@80S8b<
0`18B4P309jNWP80MgMg0`1oOglJ07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?oo
o`020?oooaX0OWin0`1mOGd208R8R0<0LG5a0`1_Kfl20:JVYP<0SXj>0P1:BTX305IFEP<0WIfM0P37
alL309>CT`80P8200`0S8b<301XJ6P80Y:BT0`2/[:`207inOP<0W9bL0`3MgMd20>GUi@<0Xj>S0`19
BDT200h>3P<01@D50P0R8R8304a<C0<0KVi^0P20P8030861P@80MWIf0`1WIfL305UIF@80CTi>0`18
B4P204I6AP<0B4Q80`1:BTX204e=C@<0Cdm?0P1>CTh304a<C0<0AdM70P11@D4303/k>`<0=cLg0P0f
=SH303/k>`80AdM70`1KFe/307AdM080SHf=0`2QXJ420:FUY@<0TI6A0`1WIfL2034a<@<02PX:0P0>
3Ph304Q8B0<0WinO0P3Jf]X30<c<c0<0Shn?0P1kNg/30:BTY080VIVI0`0n?Sh3034a<@80K6a/0`2K
Vi/20<30`0<0XJ6Q0`1CDe<20410@0<0U9BD0P2j^[X305QHF0<0L71`0P2HV9P3079bLP<0PH615`1n
OWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3ooolJ07inOP<0OGem0P28R8P3
075aL@<0Kfm_0P2VYZH308j>SP80BTY:0`1FEUH309fMW@80alO70`2CTi<20820P0<08b<S0`0J6QX2
0:BTY0<0[:b/0P1nOWh309bLW0<0gMgM0P3UiND30:>SX`<0BDU90P0>3Ph300D51@808R8R0`1<C4`3
06i^KP80P8200`21PH4207IfMP<0IfMW0`1IFET204i>CP<0B4Q80P16ATH304Q8B0<0BTY:0P1=CDd3
04m?C`80CTi>0`1<C4`304M7A`80@D510`0k>c/303Lg=`80=SHf0`0k>c/204M7A`<0Fe]K0`1dM7@2
08f=S@<0XJ6Q0P2UYJD3096AT@<0IfMW0P0a<C4300X:2P803Ph>0`18B4P309nOW`80f][J0`3<c<`3
08n?S`80Ng]k0`2TY:@209VIV@<0?Shn0`0a<C4206a/K0<0Vi^K0P30`<030:6QX@<0De=C0P10@403
09BDU080^[Zj0`1HF5P3071`L080V9RH0`1bLW830861PAL0OWin1@3oool100000040oooo000?0?oo
o`030000003oool0oooo0080oooo6P1nOWh307emO@80R8R80`1aLG4306m_K`80YZJV0`2>SXh204Y:
BP<0EUIF0`2MWId20<O7a`<0Ti>C0P20P80302<S8`<06QXJ0P2TY:@30:b/[080OWin0`2LW9`30=gM
g@80iNGU0`2SXj<304U9B@803Ph>0`051@D2028R8P<0C4a<0`1^KVh20820P0<0PH610P1fMWH306MW
I`<0FEUI0P1>CTh304Q8B080ATI60`18B4P304Y:BP80CDe=0`1?Cdl204i>CP<0C4a<0`17AdL20451
@@<0>c/k0`0g=cL203Hf=P<0>c/k0P17AdL305]KF`<0M7Ad0P2=SHd30:6QX@80YJFU0`2ATI4306MW
I`80<C4a0`0:2PX200h>3P<0B4Q80`2OWil20=[JfP<0c<c<0`2?Shl207]kN`<0Y:BT0P2IVIT303hn
?P<0<C4a0P1/K6`309^KV`80`<300`2QXJ4305=CD`80@4100`2DU9@20;Zj^P<0F5QH0`1`L70209RH
V0<0LW9b0`21PH4G07inOPD0oooo0@0000010?ooo`003`3oool2000000<0oooo5@1nOWh207moO`<0
O7al1@22PX8306IVIP<0SHf=0P2UYJD306ATI080ATI60`20P8030;fm_@80/k>c0`1bLW8205aLG0<0
0@410`1bLW820<S8b0<0UYJF0P27QhL30;no_`<0gmoO0P2a/K4305MGE`<03`l?0P00000301PH6080
BDU90`1dM7@308f=S@80TY:B0`29RHT207]kN`<0KVi^0`1UIFD2069RHP<0IFEU0P1[Jf/307=cL`<0
Ng]k0P22PX8308J6QP80QhN70`24Q8@307emO@80Lg=c0`1XJ6P305aLG080DE5A0`1;Bd/204a<C0<0
EEEE0`1WIfL20820P0<0Vi^K0P2][Jd30:j^[P<0U9BD0P1QHF4302LW9`800`<30`0C4a<305UIF@80
[Zj^0`3Cdm<30:n_[`80Ogmo0`2BTY820;Jf]P<0N7Qh0`0/;2`204Q8B0<0NGUi0P2/[:`30;jn_P<0
PX:20P0n?Sh305eMG@80^;Rh0`2ATI4304Y:BP80W9bL0`1fMWH307moOaL0OWin103oool200000040
oooo000?0?ooo`030000003oool0oooo0080oooo5@1nOWh207moO`<0O7al1@22PX8306IVIP<0SHf=
0P2UYJD306ATI080ATI60`20P8030;fm_@80/k>c0`1bLW8205aLG0<00@410`1bLW820<S8b0<0UYJF
0P27QhL30;no_`<0gmoO0P2a/K4305MGE`<03`l?0P00000301PH6080BDU90`1dM7@308f=S@80TY:B
0`29RHT207]kN`<0KVi^0`1UIFD2069RHP<0IFEU0P1[Jf/307=cL`<0Ng]k0P22PX8308J6QP80QhN7
0`24Q8@307emO@80Lg=c0`1XJ6P305aLG080DE5A0`1;Bd/204a<C0<0EEEE0`1WIfL20820P0<0Vi^K
0P2][Jd30:j^[P<0U9BD0P1QHF4302LW9`800`<30`0C4a<305UIF@80[Zj^0`3Cdm<30:n_[`80Ogmo
0`2BTY820;Jf]P<0N7Qh0`0/;2`204Q8B0<0NGUi0P2/[:`30;jn_P<0PX:20P0n?Sh305eMG@80^;Rh
0`2ATI4304Y:BP80W9bL0`1fMWH307moOaL0OWin1@3oool100000040oooo000?0?ooo`030000003o
ool0oooo0080oooo5`1nOWh307emO@<0QhN71@1bLW830:BTY080QXJ60`19BDT2069RHP<0ZZZZ0`32
`/820861P@<0JFUY0P0R8R83034a<@<0`/;20P2l_;`308F5Q@80WinO0`3?cll30;bl_080JFUY0`0G
5aL3000000803`l?0`15ADD207UiN@<0VYZJ0`2TY:@209fMW@<0Shn?0P21PH4307UiN@<0N7Qh0P1m
OGd308R8R080U9BD0`2QXJ430:f][@80][Jf0`2k^k/20;fm_@<0^[Zj0`2c/k<20:RXZ0<0VIVI0`28
R8P207MgM`<0IfMW0P1MGEd305YJFP<0HF5Q0P1cLg<308f=S@80ZJVY0`2k^k/30;Ng]`80UIFE0`1J
FUX201`L70<000000`0N7Qh206i^KP<0^;Rh0`31`L4209>CT`<0PH610P2_[jl30;2`/0<0DU9B0P0a
<C4305YJFP80R8R80`2l_;`30:j^[P80H61P0`0m?Cd208^;R`<0^k^k0`1LG5`2075aL@<0U9BD0`1d
M7@20861PAD0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo5`1nOWh3
07emO@<0QhN71@1bLW830:BTY080QXJ60`19BDT2069RHP<0ZZZZ0`32`/820861P@<0JFUY0P0R8R83
034a<@<0`/;20P2l_;`308F5Q@80WinO0`3?cll30;bl_080JFUY0`0G5aL3000000803`l?0`15ADD2
07UiN@<0VYZJ0`2TY:@209fMW@<0Shn?0P21PH4307UiN@<0N7Qh0P1mOGd308R8R080U9BD0`2QXJ43
0:f][@80][Jf0`2k^k/20;fm_@<0^[Zj0`2c/k<20:RXZ0<0VIVI0`28R8P207MgM`<0IfMW0P1MGEd3
05YJFP<0HF5Q0P1cLg<308f=S@80ZJVY0`2k^k/30;Ng]`80UIFE0`1JFUX201`L70<000000`0N7Qh2
06i^KP<0^;Rh0`31`L4209>CT`<0PH610P2_[jl30;2`/0<0DU9B0P0a<C4305YJFP80R8R80`2l_;`3
0:j^[P80H61P0`0m?Cd208^;R`<0^k^k0`1LG5`2075aL@<0U9BD0`1dM7@20861PAD0OWin1@3oool1
00000040oooo000?0?ooo`030000003oool0oooo0080oooo5`1nOWh307emO@<0QhN71@1bLW830:BT
Y080QXJ60`19BDT2069RHP<0ZZZZ0`32`/820861P@<0JFUY0P0R8R83034a<@<0`/;20P2l_;`308F5
Q@80WinO0`3?cll30;bl_080JFUY0`0G5aL3000000803`l?0`15ADD207UiN@<0VYZJ0`2TY:@209fM
W@<0Shn?0P21PH4307UiN@<0N7Qh0P1mOGd308R8R080U9BD0`2QXJ430:f][@80][Jf0`2k^k/20;fm
_@<0^[Zj0`2c/k<20:RXZ0<0VIVI0`28R8P207MgM`<0IfMW0P1MGEd305YJFP<0HF5Q0P1cLg<308f=
S@80ZJVY0`2k^k/30;Ng]`80UIFE0`1JFUX201`L70<000000`0N7Qh206i^KP<0^;Rh0`31`L4209>C
T`<0PH610P2_[jl30;2`/0<0DU9B0P0a<C4305YJFP80R8R80`2l_;`30:j^[P80H61P0`0m?Cd208^;
R`<0^k^k0`1LG5`2075aL@<0U9BD0`1dM7@20861PAD0OWin1@3oool100000040oooo000?0?ooo`03
0000003oool0oooo0080oooo4P1nOWh307moO`80O7al0`22PX830861P@80JFUY0`2?Shl309nOW`80
GemO0`1=CDd208j>SP<0_kno0`2UYJD205aLG0<0De=C0P0A4A43096AT@<0gMgM0P2JVYX308J6QP80
]KFe0`2n_[h307inOP809RHV0`00000300L71`80?clo0`1mOGd20:NWY`<0][Jf0`2`/;020:2PX0<0
TI6A0P27QhL308J6QP<0SHf=0P2JVYX30:VYZ@80^KVi0`37alL30=CDe080gMgM0`3Shn<20>GUi@<0
hn?S0`3MgMd20=;BdP<0a<C40`2c/k<209jNWP<0RHV90P1fMWH306QXJ0<0Hf=S0P1YJFT307alO080
VIVI0`2g]kL30<S8b080_kno0`2DU9@2055AD@<051@D0`020P82030`<0<0Ph>30`2i^KT20:RXZ0<0
Ogmo0P2EUID30<G5a@<0TI6A0P0j>SX30451@@80IVIV0`2NWYh30<71`@80TI6A0`14A4@205EEE@<0
/k>c0`2DU9@204a<C0<0VYZJ0`1fMWH20820P1D0OWin1@3oool100000040oooo000?0?ooo`030000
003oool0oooo0080oooo4P1nOWh307moO`80O7al0`22PX830861P@80JFUY0`2?Shl309nOW`80GemO
0`1=CDd208j>SP<0_kno0`2UYJD205aLG0<0De=C0P0A4A43096AT@<0gMgM0P2JVYX308J6QP80]KFe
0`2n_[h307inOP809RHV0`00000300L71`80?clo0`1mOGd20:NWY`<0][Jf0`2`/;020:2PX0<0TI6A
0P27QhL308J6QP<0SHf=0P2JVYX30:VYZ@80^KVi0`37alL30=CDe080gMgM0`3Shn<20>GUi@<0hn?S
0`3MgMd20=;BdP<0a<C40`2c/k<209jNWP<0RHV90P1fMWH306QXJ0<0Hf=S0P1YJFT307alO080VIVI
0`2g]kL30<S8b080_kno0`2DU9@2055AD@<051@D0`020P82030`<0<0Ph>30`2i^KT20:RXZ0<0Ogmo
0P2EUID30<G5a@<0TI6A0P0j>SX30451@@80IVIV0`2NWYh30<71`@80TI6A0`14A4@205EEE@<0/k>c
0`2DU9@204a<C0<0VYZJ0`1fMWH20820P1D0OWin1@3oool100000040oooo000?0?ooo`030000003o
ool0oooo0080oooo5@1nOWh207emO@<0QXJ60`1cLg<207EeM@<0Xj>S0`1oOgl204U9B@<0KVi^0P2b
/[830;Rh^0<0Kfm_0P1IFET302/[:`80CDe=0`3Ng]h30<?3``80P8200`2EUID20;Jf]P<0T92@0`0k
>c/2000000<00@410`0h>3P207emO@<0/K6a0P36a/H30<;2`P<0/;2`0P2LW9`3092@T080SXj>0`2D
U9@30:6QX@80/K6a0`31`L420=7Ad@<0g]kN0`3Xj>P20?3`l0<0mOGe0P3gmoL30?KfmP<0l_;b0P3[
jn/30>3Ph0<0dM7A0P2o_kl30:VYZ@80Ti>C0`1mOGd306e]K@80IfMW0`1^KVh208B4Q0<0Y:BT0`34
a<@20=CDe0<0aLG50P2@T90304M7A`<03@d=0P0;2`/304M7A`<0UIFE0P2_[jl308b<S080OGem0`2d
];@30<C4a080JFUY0`0i>CT204e=C@<0MWIf0`2c/k<20;Ng]`<0KFe]0P0l?3`308:2PP<0^k^k0P1O
Gel3079bLP<0TI6A0P1fMWH30820P180OWin1@3oool100000040oooo000?0?ooo`030000003oool0
oooo0080oooo5@1nOWh207emO@<0QXJ60`1cLg<207EeM@<0Xj>S0`1oOgl204U9B@<0KVi^0P2b/[83
0;Rh^0<0Kfm_0P1IFET302/[:`80CDe=0`3Ng]h30<?3``80P8200`2EUID20;Jf]P<0T92@0`0k>c/2
000000<00@410`0h>3P207emO@<0/K6a0P36a/H30<;2`P<0/;2`0P2LW9`3092@T080SXj>0`2DU9@3
0:6QX@80/K6a0`31`L420=7Ad@<0g]kN0`3Xj>P20?3`l0<0mOGe0P3gmoL30?KfmP<0l_;b0P3[jn/3
0>3Ph0<0dM7A0P2o_kl30:VYZ@80Ti>C0`1mOGd306e]K@80IfMW0`1^KVh208B4Q0<0Y:BT0`34a<@2
0=CDe0<0aLG50P2@T90304M7A`<03@d=0P0;2`/304M7A`<0UIFE0P2_[jl308b<S080OGem0`2d];@3
0<C4a080JFUY0`0i>CT204e=C@<0MWIf0`2c/k<20;Ng]`<0KFe]0P0l?3`308:2PP<0^k^k0P1OGel3
079bLP<0TI6A0P1fMWH30820P180OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo
0080oooo5@1nOWh207emO@<0QXJ60`1cLg<207EeM@<0Xj>S0`1oOgl204U9B@<0KVi^0P2b/[830;Rh
^0<0Kfm_0P1IFET302/[:`80CDe=0`3Ng]h30<?3``80P8200`2EUID20;Jf]P<0T92@0`0k>c/20000
00<00@410`0h>3P207emO@<0/K6a0P36a/H30<;2`P<0/;2`0P2LW9`3092@T080SXj>0`2DU9@30:6Q
X@80/K6a0`31`L420=7Ad@<0g]kN0`3Xj>P20?3`l0<0mOGe0P3gmoL30?KfmP<0l_;b0P3[jn/30>3P
h0<0dM7A0P2o_kl30:VYZ@80Ti>C0`1mOGd306e]K@80IfMW0`1^KVh208B4Q0<0Y:BT0`34a<@20=CD
e0<0aLG50P2@T90304M7A`<03@d=0P0;2`/304M7A`<0UIFE0P2_[jl308b<S080OGem0`2d];@30<C4
a080JFUY0`0i>CT204e=C@<0MWIf0`2c/k<20;Ng]`<0KFe]0P0l?3`308:2PP<0^k^k0P1OGel3079b
LP<0TI6A0P1fMWH30820P180OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080
oooo4P1nOWh307emO@D0PH610`1[Jf/2092@T0<0VYZJ0`1KFe/205EEE@<0VYZJ0P2m_Kd309BDU0<0
BTY:0P1BDU8302LW9`80[jn_0`3Zj^X309>CT`80NWYj0`2QXJ4209^KV`<0E5AD0`0<30`2000000<0
;bl_0`1jNWX20;Rh^0<0e=CD0P3AdM430;bl_0<0Y:BT0P2DU9@308j>SP80TY:B0`2NWYh30:f][@80
_Kfm0`3;bl/20=OGe`<0h>3P0`3Vi^H20>[ZjP<0kNg]203_knl20>c/k0<0inOW0`3Ng]h20=7Ad@<0
`<300P2[Zj/309>CT`<0OGem0P1/K6`306MWI`80L71`0`2:RXX30:n_[`80dM7A0`3Ogml20<O7a`<0
RHV90`0k>c/200/;2`<06a/K0`1PH60209jNWP<0VYZJ0P1fMWH308n?S`<0c/k>0P2XZ:P304i>CP80
@T920`1FEUH308j>SP80_kno0`2NWYh204a<C0<0Cdm?0`2][Jd209NGU`<0Cdm?0`2IVIT207EeM@<0
P8204P1nOWh50?ooo`4000000@3oool000l0oooo0P0000030?oooa80OWin0`1mOGd50861P@<0Jf][
0P2@T90309ZJVP<0Fe]K0P1EEED309ZJVP80_Kfm0`2DU9@304Y:BP80DU9B0`0W9bL20:n_[`<0j^[Z
0`2CTi<207YjNP<0XJ6Q0P2KVi/305ADE0<030`<0P00000302l_;`<0NWYj0P2h^;P30=CDe080dM7A
0`2l_;`30:BTY080U9BD0`2>SXh209:BTP<0WYjN0`2][Jd20;fm_@<0bl_;0P3GemL30>3Ph0<0i^KV
0P3Zj^X30>g]k@P0kno_0P3/k>`30>OWi`<0g]kN0P3AdM430<30`080Zj^[0`2CTi<307emO@80K6a/
0`1WIfL2071`L0<0RXZ:0`2_[jl20=7Ad@<0gmoO0P37alL308V9R@<0>c/k0P0;2`/301/K6`<0H61P
0P2NWYh309ZJVP80MWIf0`2?Shl30<k>cP80Z:RX0`1>CTh20492@P<0EUIF0`2>SXh20;no_`<0WYjN
0P1<C4`304m?C`<0[Jf]0P2GUiL304m?C`<0VIVI0P1eMGD30820P180OWin103oool200000040oooo
000?0?ooo`030000003oool0oooo0080oooo4P1nOWh307emO@D0PH610`1[Jf/2092@T0<0VYZJ0`1K
Fe/205EEE@<0VYZJ0P2m_Kd309BDU0<0BTY:0P1BDU8302LW9`80[jn_0`3Zj^X309>CT`80NWYj0`2Q
XJ4209^KV`<0E5AD0`0<30`2000000<0;bl_0`1jNWX20;Rh^0<0e=CD0P3AdM430;bl_0<0Y:BT0P2D
U9@308j>SP80TY:B0`2NWYh30:f][@80_Kfm0`3;bl/20=OGe`<0h>3P0`3Vi^H20>[ZjP<0kNg]203_
knl20>c/k0<0inOW0`3Ng]h20=7Ad@<0`<300P2[Zj/309>CT`<0OGem0P1/K6`306MWI`80L71`0`2:
RXX30:n_[`80dM7A0`3Ogml20<O7a`<0RHV90`0k>c/200/;2`<06a/K0`1PH60209jNWP<0VYZJ0P1f
MWH308n?S`<0c/k>0P2XZ:P304i>CP80@T920`1FEUH308j>SP80_kno0`2NWYh204a<C0<0Cdm?0`2]
[Jd209NGU`<0Cdm?0`2IVIT207EeM@<0P8204P1nOWh50?ooo`4000000@3oool000l0oooo00<00000
0?ooo`3oool00P3ooolB07inOP<0OGem0P25QHD307AdM0<0MgMg0P2PX:0307UiN@<0C4a<0P1jNWX3
0;Ng]`80[:b/0`1NGUh3051@D080>SXj0`1YJFT20?;blP<0_[jn0`1`L70208B4Q0<0Vi^K0P1/K6`3
024Q8@<00P820P0W9bL307AdM0<0^k^k0P3Ph>030=oOg`80alO70`2ZZZX309>CT`80RHV90`2:RXX2
09BDU0<0XZ:R0`2`/;020;bl_0<0aLG50P3:b/X30<g=c@D0c/k>1@3?cll30=7Ad@<0dm?C0P3Fe]H3
0=OGe`<0e]KF0P3AdM430<S8b080^KVi0`2UYJD308j>SP80N7Qh0`1WIfL2069RHP<0Kfm_0`2>SXh2
0;Rh^0<0gMgM0P3Vi^H30<G5a@<0OGem0P0`<30300h>3P<0<C4a0P1gMgL309^KV`80Ogmo0`1aLG43
0:n_[`80d];B0`21PH4504M7A`<0I6AT0P2XZ:P30;fm_@80NWYj0`0m?Cd307YjNP80^[Zj0`1RHV83
07AdM080SXj>0`1hN7P207moOa00OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo
0080oooo4P1nOWh307emO@80QHF50`1dM7@307MgM`80X:2P0`1iNGT304a<C080NWYj0`2g]kL20:b/
[0<0GUiN0`1@D50203Xj>P<0JFUY0P3bl_830;jn_P<0L71`0P24Q8@309^KV`80K6a/0`0Q8B430082
0P809bLW0`1dM7@30;^k^`80h>3P0`3Ogml20<O7a`<0ZZZZ0`2CTi<208V9R@<0RXZ:0P2DU9@30::R
XP<0/;2`0P2l_;`30<G5a@80b/[:0`3=cLd50<k>cPD0clo?0`3AdM430=?Cd`80e]KF0`3GemL30=KF
eP80dM7A0`38b<P20;Vi^@<0YJFU0`2>SXh207QhN0<0IfMW0P1RHV8306m_K`<0SXj>0P2h^;P30=gM
g@80i^KV0`35aLD307emO@80<30`0`0>3Ph3034a<@80MgMg0`2KVi/207moO`<0LG5a0`2_[jl20=;B
dP<0PH611@17AdL306ATI080Z:RX0`2m_Kd207YjNP<0?Cdm0`1jNWX20;Zj^P<0HV9R0`1dM7@208j>
SP<0N7Qh0P1oOgl@07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oooa00
OWin0P1mOGd30861P@80P8200`1]KFd3092@T080UIFE0`1HF5P305eMG@80Xj>S0`2h^;P208>3P`<0
?Shn0`1HF5P20451@@<0b<S80P3/k>`308:2PP<0IfMW0P2<S8`307moO`80>c/k0`0;2`/301lO7`80
JFUY0`2i^KT30>OWi`80jn_[0`3AdM420:f][@<0Shn?0`20P80207inOP<0QXJ60P2CTi<309nOW`<0
Z:RX0P2^[Zh30:n_[`80[Jf]0`2YZJT30:FUY@80XJ6Q1@2OWil30::RXP<0YZJV0P2][Jd30;Bd]0<0
^k^k1@2o_kl20;Zj^P<0[jn_0`2LW9`208J6QP<0Kfm_0P1NGUh305aLG0<0KFe]0P2BTY830<71`@80
inOW0`3Zj^X30;jn_P80Kfm_0`0X:2P301TI6@80Bd];0`27QhL208^;R`<0J6QX0`21PH420<k>cP<0
^k^k0P1TI6@304Q8B0<0BTY:0P1mOGd30;Zj^P80ZJVY0`1EEED304Y:BP80Z:RX0`2IVIT3055AD@80
V9RH0`1eMGD20820P100OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo
401nOWh207emO@<0PH610P20P80306e]K@<0T92@0P2EUID305QHF0<0GEeM0P2SXj<30;Rh^080Ph>3
0`0n?Sh305QHF080@D510`38b<P20>c/k0<0PX:20`1WIfL208b<S0<0Ogmo0P0k>c/300/;2`<07alO
0P1YJFT30;Vi^@<0inOW0P3[jn/30=7Ad@80[Jf]0`2?Shl30820P080OWin0`26QXH209>CT`<0WinO
0`2XZ:P20:j^[P<0[jn_0P2][Jd30:VYZ@<0YJFU0P2QXJ4509nOW`<0XZ:R0`2VYZH20:f][@<0];Bd
0`2k^k/50;no_`80^[Zj0`2_[jl309bLW080QXJ60`1_Kfl205iNGP<0G5aL0`1]KFd209:BTP<0`L71
0P3WinL30>[ZjP<0_[jn0P1_Kfl302PX:0<06ATI0P1;Bd/308N7Q`80Rh^;0`1XJ6P30861P@80c/k>
0`2k^k/206ATI0<0B4Q80`1:BTX207emO@<0^[Zj0P2YZJT305EEE@<0BTY:0P2XZ:P309VIV@<0DE5A
0P2HV9P307EeM@80P820401nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3o
ool@07inOP80OGem0`21PH420820P0<0KFe]0`2@T90209FEU@<0F5QH0`1MGEd20:>SX`<0^;Rh0P23
Ph<303hn?P<0F5QH0P11@D430<S8b080k>c/0`22PX8306MWI`80S8b<0`1oOgl203/k>`<02`/;0`0O
7al206UYJ@<0^KVi0`3WinL20>_[j`<0dM7A0P2][Jd308n?S`<0P8200P1nOWh308J6QP80Ti>C0`2O
Wil30:RXZ080[Zj^0`2_[jl20:f][@<0ZJVY0`2UYJD20:6QX@D0WinO0`2RXZ830:JVYP80[Jf]0`2d
];@30;^k^`D0_kno0P2j^[X30:n_[`<0W9bL0P26QXH306m_K`80GUiN0`1LG5`306e]K@80TY:B0`31
`L420>OWi`<0j^[Z0`2n_[h206m_K`<0:2PX0`0I6AT204];B`<0QhN70P2;Rh/306QXJ0<0PH610P3>
c/h30;^k^`80I6AT0`18B4P304Y:BP80OGem0`2j^[X20:VYZ@<0EEEE0`1:BTX20:RXZ0<0VIVI0`1A
DE4209RHV0<0MGEe0P20P80@07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`02
0?oooa00OWin0P1mOGd308B4Q080MGEe0`1iNGT309jNWP80MGEe0`1?Cdl308F5Q@80^KVi0`2MWId2
051@D0<0CDe=0`1?Cdl208B4Q0<0o?cl0P2_[jl305UIF@<0LG5a0P26QXH305IFEPD071`L0`1LG5`2
0;6a/@<0j^[Z0`3emOD20=WIf@<0[jn_0P2:RXX307AdM0<0Kfm_0P1eMGD30861P@80S8b<0`2DU9@3
09NGU`80U9BD0`2>SXh208F5Q@<0O7al0`1dM7@206e]K@D0JFUY0`1[Jf/3075aL@80NWYj0`26QXH3
09:BTP80WYjN0`2XZ:P20:f][@<0[:b/0`2SXj<209:BTP<0Ng]k0P1TI6@605=CD`80JVYZ0`2FUYH2
0<_;b`<0kno_0`3Zj^X20;>c/`<0H61P0`0U9BD202/[:`<0IFEU0P2:RXX3079bLP<0GemO0P2RXZ83
0=SHf080UiNG0`1IFET304M7A`80EEEE0`2JVYX20;no_`<0QhN70`0o?cl207=cL`<0^;Rh0`1TI6@2
07EeM@<0Rh^;0P1iNGT307moO`d0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo
0080oooo401nOWh207emO@<0Q8B40P1eMGD307UiN@<0WYjN0P1eMGD304m?C`<0QHF50P2i^KT309fM
W@80D51@0`1=CDd304m?C`80Q8B40`3lo?`20:n_[`<0FEUI0`1aLG4208J6QP<0EUIF1@0L71`305aL
G080/K6a0`3Zj^X30?Gem@80fMWI0`2_[jl208Z:RP<0M7Ad0`1_Kfl207EeM@<0PH610P2<S8`309BD
U0<0UiNG0P2DU9@308j>SP80QHF50`1lO7`307AdM080KFe]1@1YJFT306][J`<0LG5a0P1jNWX308J6
QP<0TY:B0P2NWYh30:RXZ080[Jf]0`2/[:`30:>SX`80TY:B0`1kNg/206ATI0H0De=C0P1ZJVX309JF
UP80bl_;0`3_knl30>[ZjP80/k>c0`1PH60302DU9@80:b/[0`1UIFD208Z:RP<0LW9b0`1OGel20::R
XP<0f=SH0P2GUiL305UIF@<0AdM70P1EEED309ZJVP80_kno0`27QhL303lo?`80Lg=c0`2h^;P306AT
I080MGEe0`2;Rh/207UiN@<0Ogmo3@1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3o
ool00P3oool=07inOP<0OGem1@20P80206m_K`<0T92@0`2ATI4205MGE`<0IVIV0`2ZZZX20;2`/0<0
LW9b0P0h>3P306ATI0<0Fe]K0P3Jf]X30>CTi080JFUY0`1CDe<307alO080KVi^0`0d=3@201hN7P<0
Cdm?0`2TY:@20>SXj0<0no_k0`3Rh^820;:b/P<0QHF50P1XJ6P305mOG`<0I6AT0P1`L70307]kN`80
PX:20`24Q8@307moO`80MGEe0`1YJFT205aLG0<0D51@0`15ADD203dm?@<0>3Ph0P0g=cL303Xj>P<0
@4100P19BDT305IFEP<0IVIV0P1fMWH308N7Q`80UIFE0`2NWYh30:2PX080VIVI0`28R8P2071`L0<0
EeMG0`18B4P204];B`<0J6QX0P2KVi/30=CDe0<0mOGe0P3UiND30:>SX`<0DE5A0P0Y:BT304=3@`80
NGUi0`1oOgl305UIF@80KFe]0`36a/H20<S8b0<0NWYj0`1DE5@204A4A0<0KFe]0P2c/k<30;:b/P<0
GemO0P16ATH30:>SX`<0VYZJ0P1DE5@309JFUP80MWIf0`20P80=07inOPD0oooo0@0000010?ooo`00
0`3oool300000080oooo0P0000050?ooo`030000003oool0oooo0080oooo3@1nOWh307emO@D0P820
0P1_Kfl3092@T0<0TI6A0P1GEeL306IVIP<0ZZZZ0P2`/;03079bLP80>3Ph0`1TI6@305]KF`80f][J
0`3Ti>@206UYJ@<0De=C0`1lO7`206i^KP<0=3@d0P0N7Qh304m?C`<0Y:BT0P3Xj>P30?_kn`<0h^;R
0P2b/[8308F5Q@80J6QX0`1OGel306ATI080L71`0`1kNg/208:2PP<0Q8B40`1oOgl207EeM@<0JFUY
0P1LG5`3051@D0<0ADE50P0m?Cd303Ph>080=cLg0`0j>SX30410@080BDU90`1FEUH306IVIP80MWIf
0`27QhL209FEU@<0WYjN0`2PX:0209VIV@<0R8R80P1`L70305MGE`<0B4Q80P1;Bd/306QXJ080Vi^K
0`3De=@30?Gem@80iNGU0`2SXj<3055AD@80:BTY0`13@d<207UiN@<0Ogmo0`1IFET206e]K@<0a/K6
0P38b<P307YjNP<0E5AD0P14A4@306e]K@80/k>c0`2b/[8305mOG`80ATI60`2SXj<309ZJVP80E5AD
0`2FUYH207IfMP<0P8203@1nOWh50?ooo`4000000@3oool000@0oooo00@000000?ooo`3oool00000
0P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo3@1nOWh307emO@D0P820
0P1_Kfl3092@T0<0TI6A0P1GEeL306IVIP<0ZZZZ0P2`/;03079bLP80>3Ph0`1TI6@305]KF`80f][J
0`3Ti>@206UYJ@<0De=C0`1lO7`206i^KP<0=3@d0P0N7Qh304m?C`<0Y:BT0P3Xj>P30?_kn`<0h^;R
0P2b/[8308F5Q@80J6QX0`1OGel306ATI080L71`0`1kNg/208:2PP<0Q8B40`1oOgl207EeM@<0JFUY
0P1LG5`3051@D0<0ADE50P0m?Cd303Ph>080=cLg0`0j>SX30410@080BDU90`1FEUH306IVIP80MWIf
0`27QhL209FEU@<0WYjN0`2PX:0209VIV@<0R8R80P1`L70305MGE`<0B4Q80P1;Bd/306QXJ080Vi^K
0`3De=@30?Gem@80iNGU0`2SXj<3055AD@80:BTY0`13@d<207UiN@<0Ogmo0`1IFET206e]K@<0a/K6
0P38b<P307YjNP<0E5AD0P14A4@306e]K@80/k>c0`2b/[8305mOG`80ATI60`2SXj<309ZJVP80E5AD
0`2FUYH207IfMP<0P8203@1nOWh50?ooo`4000000@3oool000030?ooo`000000000000<0000000D0
oooo0000003oool0oooo000000040?ooo`<000000P3oool=07inOP<0OGem0P23Ph<307IfMP80NWYj
0`2LW9`3075aL@80De=C0`2>SXh30;Ng]`80SXj>0`16ATH2051@D0<0IVIV0`2IVIT20?gmo@<0UiNG
0P11@D4306=SH`<0NWYj0P1@D50302TY:@80A4A40`2DU9@30>7Qh@80oooo0`3YjNT30;Jf]P80P820
0`1LG5`204m?C`<0De=C0`1PH60206e]K@<0MGEe0P1fMWH3075aL@<0IVIV0P1HF5P304U9B@80>SXj
0`0^;Rh302<S8`8071`L0`0G5aL201DE5@<05aLG0`0L71`202@T90<0<30`0`0n?Sh2051@D0<0I6AT
0P1gMgL308R8R0<0U9BD0P2HV9P3096AT@80Ogmo0`1UIFD304];B`80?3`l0`13@d<206MWI`<0XZ:R
0`3MgMd20?Ogm`<0fm_K0`2@T90204I6AP<0=CDe0P1NGUh30861P@<0J6QX0P1=CDd308n?S`80emOG
0`2YZJT306e]K@80C4a<0`1:BTX208^;R`<0_kno0`2BTY8204=3@`<0KFe]0`2g]kL206IVIP<0MgMg
0P28R8P307]kN`<0Ogmo2P1nOWh40?ooo`8000000@3oool000050?ooo`000000oooo0?ooo`000000
0P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0080oooo3@1nOWh307emO@80
Ph>30`1fMWH207YjNP<0W9bL0`1aLG4205=CD`<0SXj>0`2g]kL208j>SP<0ATI60P1@D50306IVIP<0
VIVI0P3moOd309NGU`80@D510`1SHf<307YjNP80D51@0`0Y:BT204A4A0<0U9BD0`3QhN420?ooo`<0
jNWY0`2f][H20820P0<0G5aL0P1?Cdl305=CD`<0H61P0P1]KFd307EeM@80MWIf0`1aLG4306IVIP80
F5QH0`19BDT203Xj>P<0;Rh^0`0S8b<201`L70<05aLG0P0E5AD301LG5`<071`L0P0T92@3030`<0<0
?Shn0P1@D50306ATI080MgMg0`28R8P309BDU080V9RH0`2ATI4207moO`<0IFEU0`1;Bd/203`l?0<0
@d=30P1WIfL30::RXP<0gMgM0P3gmoL30=_Kf`<0T92@0P16ATH303De=@80GUiN0`21PH4306QXJ080
CDe=0`2?Shl20=OGe`<0ZJVY0`1]KFd204a<C0<0BTY:0P2;Rh/30;no_`<0TY:B0P13@d<306e]K@<0
]kNg0P1VIVH307MgM`80R8R80`1kNg/307moO`X0OWin1@3oool100000040oooo00020?ooo`030000
003oool000000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`d0
OWin0`1mOGd208>3P`<0MWIf0P1jNWX309bLW0<0LG5a0P1CDe<308j>SP<0]kNg0P2>SXh304I6AP80
D51@0`1VIVH309VIV@80oOgm0`2GUiL20451@@<0Hf=S0`1jNWX2051@D0<0:BTY0P14A4@309BDU0<0
hN7Q0P3oool30>WYj@<0][Jf0P20P80305aLG080Cdm?0`1CDe<3061PH080KFe]0`1eMGD207IfMP<0
LG5a0`1VIVH205QHF0<0BDU90P0j>SX302h^;P<08b<S0P0L71`301LG5`805ADE0`0G5aL301`L7080
92@T0`0`<30303hn?P80D51@0`1TI6@207MgM`<0R8R80`2DU9@209RHV0<0TI6A0P1oOgl306EUI@<0
Bd];0P0l?3`304=3@`80IfMW0`2RXZ830=gMg@80moOg0`3Kfm/3092@T080ATI60`0e=CD205iNGP<0
PH610`1XJ6P204e=C@<0Shn?0P3GemL30:VYZ@<0KFe]0P1<C4`304Y:BP80Rh^;0`2o_kl309:BTP80
@d=30`1]KFd30;Ng]`80IVIV0`1gMgL208R8R0<0Ng]k0`1oOgl:07inOPD0oooo0@0000010?ooo`00
0`3oool2000000<0oooo0P0000050?ooo`030000003oool0oooo0080oooo2P1nOWh307emO@D0P820
0`1`L702092@T0<0SHf=0`1GEeL206i^KP<0[Zj^0`2VYZH206ATI0<0=cLg0P1dM7@3079bLP<0h^;R
0P3Cdm<304a<C080@T920`1eMGD306UYJ@80?3`l0`0o?cl20861P@<0dm?C0`3no_h20?3`l0<0^k^k
0`1nOWh2059BDP<0@4100P13@d<3059BDP<0HV9R0P1]KFd3071`L080JVYZ0`1PH603055AD@80@T92
0`0d=3@202LW9`<07AdM0`0E5AD200l?3`<030`<1@0:2PX300d=3@804Q8B0`0J6QX302DU9@80=3@d
0`16ATH205YJFP<0Kfm_0`22PX82092@T0<0U9BD0P2=SHd307QhN0<0G5aL0P0o?cl3030`<080?3`l
0`1YJFT30:ZZZP80iNGU0`3emOD30<_;b`80O7al0`12@T8204Q8B0<0MGEe0`1jNWX204e=C@<0F5QH
0P2f][H30<k>cP<0Shn?0P1TI6@30492@P80GUiN0`2XZ:P30;Rh^080JFUY0`14A4@309jNWP80VYZJ
0`1FEUH209FEU@<0MWIf0`20P80:07inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?oo
o`020?ooo`X0OWin0`1mOGd50820P0<0L71`0P2@T90308f=S@<0EeMG0P1^KVh30:j^[P<0YZJV0P1T
I6@303Lg=`80M7Ad0`1bLW830>;RhP80dm?C0`1<C4`20492@P<0MGEe0`1YJFT203`l?0<0?clo0P21
PH430=?Cd`<0o_kn0P3`l?030;^k^`<0OWin0P1BDU830410@080@d=30`1BDU83069RHP80KFe]0`1`
L70206YZJP<0H61P0`1ADE420492@P<0=3@d0P0W9bL301dM7@<05ADE0P0?3`l300`<30D02PX:0`0=
3@d2018B4P<06QXJ0`0U9BD203@d=0<0ATI60P1JFUX306m_K`<0PX:20P2@T90309BDU080SHf=0`1h
N7P305aLG080?clo0`0`<30203`l?0<0JFUY0`2ZZZX20>GUi@<0mOGe0`3;bl/207alO0<0@T920P18
B4P307EeM@<0NWYj0P1=CDd305QHF080][Jf0`3>c/h308n?S`80I6AT0`12@T8205iNGP<0Z:RX0`2h
^;P206UYJ@<0A4A40`2NWYh209ZJVP<0EUIF0P2EUID307IfMP<0P8202P1nOWh50?ooo`4000000@3o
ool000l0oooo00<000000?ooo`3oool00P3oool:07inOP<0OGem0`23Ph<207IfMP<0Ng]k0P2IVIT3
06i^KP<0F5QH0P2EUID30;>c/`<0P8200P11@D4305QHF080OGem0`2WYjL30?Gem@80Ng]k0`0[:b/2
05eMG@<0NGUi0`1FEUH20492@P<0Kfm_0P31`L430?Shn0<0mOGe0P31`L4307inOP<0BTY:0P0b<S83
03De=@80ATI60`1JFUX306UYJ@80Kfm_0`1[Jf/2069RHP<0E5AD0`16ATH203Ti>@<0;bl_0P0V9RH3
024Q8@<07AdM0P0J6QX301PH60D05aLG0`0F5QH201LG5`<06QXJ0`0O7al202PX:0<0=3@d0P15ADD3
05UIF@<0KVi^0P22PX83092@T080U9BD0`2;Rh/307=cL`80De=C0`0d=3@202HV9P<0>3Ph0`1]KFd2
0;Bd]0<0jn_[0`3^k^h20;Jf]P<0JVYZ0P17AdL3065QH@<0PH610P1TI6@303dm?@80NGUi0`3>c/h3
0;Fe]@80P8200`1EEED204=3@`<0OGem0`2k^k/209bLW0<0B4Q80`1WIfL20;Fe]@<0J6QX0P1iNGT3
08J6QP<0O7al0P1oOgl807inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?oo
o`X0OWin0`1mOGd308>3P`80MWIf0`1kNg/209VIV@<0KVi^0`1HF5P209FEU@<0/k>c0`20P8020451
@@<0F5QH0P1mOGd30:NWY`<0mOGe0P1kNg/302/[:`80GEeM0`1iNGT305IFEP80@T920`1_Kfl20<71
`@<0n?Sh0`3emOD20<71`@<0OWin0`1:BTX2038b<P<0=CDe0P16ATH305YJFP<0JFUY0P1_Kfl306][
J`80HV9R0`1DE5@304I6AP80>CTi0`0_;bl202HV9P<08B4Q0`0M7Ad201XJ6P<061PH1@0G5aL301HF
5P805aLG0`0J6QX301lO7`80:2PX0`0d=3@204E5A@<0FEUI0`1^KVh208:2PP<0T92@0P2DU9@308^;
R`<0Lg=c0P1CDe<303@d=0809RHV0`0h>3P306e]K@80];Bd0`3[jn/30>k^kP80][Jf0`1ZJVX204M7
A`<0HF5Q0`21PH4206ATI0<0?Cdm0P1iNGT30<k>cP<0]KFe0P20P80305EEE@80@d=30`1mOGd30;^k
^`80W9bL0`18B4P306MWI`80]KFe0`1XJ6P207UiN@<0QXJ60`1lO7`207moO`P0OWin1@3oool10000
0040oooo000?0?ooo`030000003oool0oooo0080oooo2P1nOWh307emO@<0Ph>30P1fMWH307]kN`80
VIVI0`1^KVh305QHF080UIFE0`2c/k<30820P080@D510`1HF5P207emO@<0YjNW0`3emOD207]kN`<0
:b/[0P1MGEd307UiN@<0EUIF0P12@T8306m_K`80`L710`3hn?P30?Gem@80`L710`1nOWh304Y:BP80
<S8b0`0e=CD204I6AP<0FUYJ0`1YJFT206m_K`<0Jf][0P1RHV8305ADE0<0ATI60P0i>CT302l_;`80
9RHV0`0Q8B4301dM7@806QXJ0`0H61P501LG5`<05QHF0P0G5aL301XJ6P<07alO0P0X:2P303@d=080
ADE50`1IFET306i^KP80PX:20`2@T90209BDU0<0Rh^;0`1cLg<205=CD`<0=3@d0P0V9RH303Ph>0<0
KFe]0P2d];@30>_[j`<0k^k^0P2f][H306YZJP80AdM70`1QHF430861P@80I6AT0`0m?Cd207UiN@<0
c/k>0`2e]KD20820P0<0EEEE0P13@d<307emO@<0^k^k0P2LW9`304Q8B0<0IfMW0P2e]KD306QXJ080
NGUi0`26QXH307alO080Ogmo201nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool0
0P3oool:07inOPH0P8200P1bLW8308n?S`80RHV90`1GEeL307IfMP80/;2`0`2KVi/305QHF080>c/k
1@25QHD30>;RhP<0^k^k0P0a<C4303Ti>@80N7Qh0`1`L70304i>CP80HV9R0`2[Zj/20>g]k@<0n?Sh
0`39bLT208:2PP<0ADE50`0W9bL202PX:0<0?3`l0P1EEED306UYJ@<0Lg=c0P1bLW8306YZJP80GUiN
0`1BDU8304M7A`80?clo0`0k>c/203Ti>@<0>3Ph0`0i>CT203Xj>P<0>c/k0P0j>SX303Ph>0<0=SHf
0P0b<S83030`<0<0;Rh^0P0_;bl303<c<`80?3`l0`1:BTX305eMG@80LW9b0`26QXH209FEU@<0V9RH
0`2;Rh/206m_K`<0BTY:0P0Y:BT301hN7P<0=cLg0P1eMGD30;no_`<0kNg]0P3QhN430:2PX080GemO
0`1EEED307UiN@80O7al0`17AdL204=3@`<0XZ:R0`3<c<`209nOW`<0MGEe0P15ADD305=CD`<0WIfM
0P2k^k/3079bLP<0@d=30P2IVIT309ZJVP80FEUI0`2DU9@307MgM`80P820201nOWh50?ooo`400000
0@3oool000l0oooo00<000000?ooo`3oool00P3oool:07inOPH0P8200P1bLW8308n?S`80RHV90`1G
EeL307IfMP80/;2`0`2KVi/305QHF080>c/k1@25QHD30>;RhP<0^k^k0P0a<C4303Ti>@80N7Qh0`1`
L70304i>CP80HV9R0`2[Zj/20>g]k@<0n?Sh0`39bLT208:2PP<0ADE50`0W9bL202PX:0<0?3`l0P1E
EED306UYJ@<0Lg=c0P1bLW8306YZJP80GUiN0`1BDU8304M7A`80?clo0`0k>c/203Ti>@<0>3Ph0`0i
>CT203Xj>P<0>c/k0P0j>SX303Ph>0<0=SHf0P0b<S83030`<0<0;Rh^0P0_;bl303<c<`80?3`l0`1:
BTX305eMG@80LW9b0`26QXH209FEU@<0V9RH0`2;Rh/206m_K`<0BTY:0P0Y:BT301hN7P<0=cLg0P1e
MGD30;no_`<0kNg]0P3QhN430:2PX080GemO0`1EEED307UiN@80O7al0`17AdL204=3@`<0XZ:R0`3<
c<`209nOW`<0MGEe0P15ADD305=CD`<0WIfM0P2k^k/3079bLP<0@d=30P2IVIT309ZJVP80FEUI0`2D
U9@307MgM`80P820201nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3oool:
07inOPH0P8200P1bLW8308n?S`80RHV90`1GEeL307IfMP80/;2`0`2KVi/305QHF080>c/k1@25QHD3
0>;RhP<0^k^k0P0a<C4303Ti>@80N7Qh0`1`L70304i>CP80HV9R0`2[Zj/20>g]k@<0n?Sh0`39bLT2
08:2PP<0ADE50`0W9bL202PX:0<0?3`l0P1EEED306UYJ@<0Lg=c0P1bLW8306YZJP80GUiN0`1BDU83
04M7A`80?clo0`0k>c/203Ti>@<0>3Ph0`0i>CT203Xj>P<0>c/k0P0j>SX303Ph>0<0=SHf0P0b<S83
030`<0<0;Rh^0P0_;bl303<c<`80?3`l0`1:BTX305eMG@80LW9b0`26QXH209FEU@<0V9RH0`2;Rh/2
06m_K`<0BTY:0P0Y:BT301hN7P<0=cLg0P1eMGD30;no_`<0kNg]0P3QhN430:2PX080GemO0`1EEED3
07UiN@80O7al0`17AdL204=3@`<0XZ:R0`3<c<`209nOW`<0MGEe0P15ADD305=CD`<0WIfM0P2k^k/3
079bLP<0@d=30P2IVIT309ZJVP80FEUI0`2DU9@307MgM`80P820201nOWh50?ooo`4000000@3oool0
00l0oooo0P0000030?ooo`P0OWin0P1mOGd308:2PP<0MgMg0P1lO7`309NGU`80K6a/0`1MGEd309^K
V`80[Zj^0`1bLW8303lo?`80I6AT0`2ATI420:j^[P<0i>CT0`1MGEd201/K6`<0HF5Q0P23Ph<306AT
I0<0GEeM0P2DU9@30=_Kf`80m_Kf0`3Bd]8308V9R@80A4A40`0N7Qh301`L7080<c<c0`1BDU8206a/
K0<0NWYj0`1mOGd207IfMP<0K6a/0P1QHF4305UIF@<0E5AD0P1CDe<305EEE@80FUYJ0`1OGel306AT
I080J6QX1@1[Jf/306UYJ@<0I6AT0P1MGEd305IFEP<0CTi>0P17AdL504=3@`<0B4Q80`1DE5@206EU
I@<0NWYj0P2>SXh309bLW0<0WIfM0P2=SHd306a/K080@T920`0O7al301PH6080>SXj0`1oOgl30<W9
b@80jn_[0`3>c/h208V9R@<0G5aL0`1[Jf/208R8R0<0IfMW0P0b<S83069RHP<0_kno0P2j^[X309>C
T`80HV9R0`10@40306m_K`80][Jf0`2TY:@304e=C@80Hf=S0`2c/k<206UYJ@<0Ng]k0`23Ph<207em
O@P0OWin103oool200000040oooo000?0?ooo`030000003oool0oooo0080oooo201nOWh207emO@<0
PX:20`1gMgL207alO0<0UiNG0P1/K6`305eMG@<0Vi^K0P2^[Zh3079bLP<0?clo0P1TI6@3096AT@80
[Zj^0`3Ti>@305eMG@806a/K0`1QHF4208>3P`<0I6AT0`1MGEd209BDU0<0fm_K0P3fm_H30=;BdP<0
RHV90P14A4@301hN7P<071`L0P0c<c<3059BDP80K6a/0`1jNWX307emO@80MWIf0`1/K6`2065QH@<0
FEUI0`1DE5@205=CD`<0EEEE0P1JFUX305mOG`<0I6AT0P1XJ6P506][J`<0JFUY0`1TI6@205eMG@<0
EUIF0`1>CTh204M7A`D0@d=30`18B4P305ADE080IFEU0`1jNWX208j>SP<0W9bL0`2MWId208f=S@<0
K6a/0P12@T8301lO7`<061PH0P0j>SX307moO`<0bLW90P3[jn/30<k>cP80RHV90`1LG5`306][J`80
R8R80`1WIfL2038b<P<0HV9R0`2o_kl20;Zj^P<0Ti>C0P1RHV830410@0<0Kfm_0P2f][H30:BTY0<0
CDe=0P1SHf<30;>c/`80JFUY0`1kNg/308>3P`80OGem201nOWh50?ooo`4000000@3oool000l0oooo
00<000000?ooo`3oool00P3oool807inOP80Ogmo0`20P80307=cL`80SXj>0`26QXH205UIF@<0OGem
0`2`/;0208n?S`<0D51@0`13@d<209NGU`<0Ti>C0P3Jf]X30:2PX0<06QXJ0P0j>SX308B4Q080OWin
0`1TI6@30861P@80aLG50`3_knl20=WIf@<0Ti>C0`17AdL201PH60<04a<C0`0/;2`2051@D0<0LG5a
0P24Q8@308Z:RP<0QHF50P1lO7`3079bLP80Jf][0`1XJ6P306][J`80LG5a0`1iNGT208>3P`<0S8b<
0`2DU9@209^KV`<0WYjN0P2OWil309fMW@<0UiNG0P2?Shl308B4Q0<0N7Qh0P1[Jf/3061PH080EUIF
0`1BDU8305ADE080GEeM0`1^KVh208>3P`<0V9RH1@2UYJD3092@T080JFUY0`0j>SX301LG5`805ADE
0`11@D4308f=S@80dM7A0`3Shn<20;Ng]`<0N7Qh0`1UIFD208>3P`<0QXJ60P18B4P3038b<P<0Rh^;
0P34a<@30:ZZZP80QXJ60`1=CDd304Y:BP80TI6A0`2m_Kd307]kN`80@d=30`2EUID209ZJVP<0Fe]K
0`2BTY8207MgM`<0Ogmo1@1nOWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3o
ool807inOP80Ogmo0`20P80307=cL`80SXj>0`26QXH205UIF@<0OGem0`2`/;0208n?S`<0D51@0`13
@d<209NGU`<0Ti>C0P3Jf]X30:2PX0<06QXJ0P0j>SX308B4Q080OWin0`1TI6@30861P@80aLG50`3_
knl20=WIf@<0Ti>C0`17AdL201PH60<04a<C0`0/;2`2051@D0<0LG5a0P24Q8@308Z:RP<0QHF50P1l
O7`3079bLP80Jf][0`1XJ6P306][J`80LG5a0`1iNGT208>3P`<0S8b<0`2DU9@209^KV`<0WYjN0P2O
Wil309fMW@<0UiNG0P2?Shl308B4Q0<0N7Qh0P1[Jf/3061PH080EUIF0`1BDU8305ADE080GEeM0`1^
KVh208>3P`<0V9RH1@2UYJD3092@T080JFUY0`0j>SX301LG5`805ADE0`11@D4308f=S@80dM7A0`3S
hn<20;Ng]`<0N7Qh0`1UIFD208>3P`<0QXJ60P18B4P3038b<P<0Rh^;0P34a<@30:ZZZP80QXJ60`1=
CDd304Y:BP80TI6A0`2m_Kd307]kN`80@d=30`2EUID209ZJVP<0Fe]K0`2BTY8207MgM`<0Ogmo1@1n
OWh50?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3oool807inOP80Ogmo0`20P803
07=cL`80SXj>0`26QXH205UIF@<0OGem0`2`/;0208n?S`<0D51@0`13@d<209NGU`<0Ti>C0P3Jf]X3
0:2PX0<06QXJ0P0j>SX308B4Q080OWin0`1TI6@30861P@80aLG50`3_knl20=WIf@<0Ti>C0`17AdL2
01PH60<04a<C0`0/;2`2051@D0<0LG5a0P24Q8@308Z:RP<0QHF50P1lO7`3079bLP80Jf][0`1XJ6P3
06][J`80LG5a0`1iNGT208>3P`<0S8b<0`2DU9@209^KV`<0WYjN0P2OWil309fMW@<0UiNG0P2?Shl3
08B4Q0<0N7Qh0P1[Jf/3061PH080EUIF0`1BDU8305ADE080GEeM0`1^KVh208>3P`<0V9RH1@2UYJD3
092@T080JFUY0`0j>SX301LG5`805ADE0`11@D4308f=S@80dM7A0`3Shn<20;Ng]`<0N7Qh0`1UIFD2
08>3P`<0QXJ60P18B4P3038b<P<0Rh^;0P34a<@30:ZZZP80QXJ60`1=CDd304Y:BP80TI6A0`2m_Kd3
07]kN`80@d=30`2EUID209ZJVP<0Fe]K0`2BTY8207MgM`<0Ogmo1@1nOWh50?ooo`4000000@3oool0
00l0oooo00<000000?ooo`3oool00P3oool807inOP80PX:20`1hN7P307emO@80U9BD0`1ZJVX2069R
HP<0X:2P0`2WYjL206MWI`<0@D510`1aLG420:>SX`<0[Jf]0P3?cll30492@P<05ADE0P1^KVh309>C
T`D0MGEe0`2/[:`20>7Qh@<0gMgM0P2OWil304i>CP<05aLG0P0;2`/302@T90<0CTi>0P1fMWH308n?
S`80V9RH0`2EUID308^;R`80PH610`1kNg/207YjNP<0OWin0`26QXH209:BTP<0WYjN0P2[Zj/30;Jf
]P<0`<300P37alL30<_;b`80c<c<0`3:b/X30<G5a@80_Kfm0`2a/K430:BTY080U9BD0`23Ph<207Ad
M0<0IfMW0`1OGel205iNGP<0IVIV0P1gMgL308b<S0<0XZ:R0P2`/;030:f][@80Ti>C0`1VIVH3038b
<P80410@0`0F5QH304a<C080Vi^K0`3EeMD20=CDe0<0WYjN0`1_Kfl207MgM`<0U9BD0P1bLW8302h^
;P<0CDe=0P2[Zj/30;Vi^@80XJ6Q0`1aLG430410@080HV9R0`2_[jl30:^[Z`80De=C0`1PH6020;6a
/@<0JFUY0`1mOGd20861P@P0OWin1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080
oooo201nOWh208:2PP<0N7Qh0`1mOGd209BDU0<0JVYZ0P1RHV830:2PX0<0YjNW0P1WIfL30451@@<0
LG5a0P2SXj<30:f][@80clo?0`12@T8301DE5@80KVi^0`2CTi<507EeM@<0[:b/0P3QhN430=gMg@80
WinO0`1>CTh301LG5`802`/;0`0T92@304i>CP80MWIf0`2?Shl209RHV0<0UIFE0`2;Rh/20861P@<0
Ng]k0P1jNWX307inOP<0QXJ60P2BTY8309jNWP80Zj^[0`2f][H30<30`080alO70`3;bl/20<c<c0<0
b/[:0`35aLD20;fm_@<0/K6a0`2TY:@209BDU0<0Ph>30P1dM7@306MWI`<0GemO0P1NGUh306IVIP80
MgMg0`2<S8`30::RXP80/;2`0`2][Jd209>CT`<0IVIV0`0b<S82010@40<05QHF0`1<C4`209^KV`<0
eMGE0P3De=@309jNWP<0Kfm_0P1gMgL309BDU080LW9b0`0^;Rh304e=C@80Zj^[0`2i^KT20:6QX@<0
LG5a0`10@402069RHP<0[jn_0`2[Zj/205=CD`<0H61P0P2a/K4306UYJ@<0OGem0P21PH4807inOPD0
oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooo`D0OWin0`1oOgl20820P0<0MGEe
0`2=SHd208B4Q0<0Fe]K0P23Ph<30:n_[`<0Q8B40P1;Bd/304i>CP<0YZJV0P2JVYX30<_;b`80QHF5
0`0;2`/304A4A080V9RH0`2>SXh207IfMP<0UIFE0`3>c/h20=cLg0<0Zj^[0P1JFUX301TI6@<01PH6
0P0M7Ad304a<C0<0Ng]k0P2JVYX30:NWY`80Y:BT0`2JVYX308n?S`80QhN70`26QXH208Z:RP<0U9BD
0`2QXJ420:n_[`<0_Kfm0P39bLT30=CDe0<0gMgM0P3Shn<30>OWi`80j>SX0`3WinL30>?Sh`80gMgM
0`3Cdm<30<O7a`80^;Rh0`2VYZH209BDU0<0PH610`1bLW8206QXJ0<0IVIV0P1]KFd307inOP<0UYJF
0P2][Jd30;Zj^P80];Bd0`2EUID3065QH@80:RXZ0`0;2`/301/K6`80Fe]K0`2YZJT20=CDe0<0`<30
0`29RHT2079bLP<0SXj>0P2FUYH3055AD@<0:2PX0P1eMGD30;Jf]P80/;2`0`2FUYH305MGE`80A4A4
0`25QHD30;bl_080Ph>30`14A4@209:BTP<0VIVI0`1NGUh2092@T0<0N7Qh0P1oOgl307inOPD0oooo
0@0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooo`D0OWin0`1oOgl20820P0<0MGEe0`2=
SHd208B4Q0<0Fe]K0P23Ph<30:n_[`<0Q8B40P1;Bd/304i>CP<0YZJV0P2JVYX30<_;b`80QHF50`0;
2`/304A4A080V9RH0`2>SXh207IfMP<0UIFE0`3>c/h20=cLg0<0Zj^[0P1JFUX301TI6@<01PH60P0M
7Ad304a<C0<0Ng]k0P2JVYX30:NWY`80Y:BT0`2JVYX308n?S`80QhN70`26QXH208Z:RP<0U9BD0`2Q
XJ420:n_[`<0_Kfm0P39bLT30=CDe0<0gMgM0P3Shn<30>OWi`80j>SX0`3WinL30>?Sh`80gMgM0`3C
dm<30<O7a`80^;Rh0`2VYZH209BDU0<0PH610`1bLW8206QXJ0<0IVIV0P1]KFd307inOP<0UYJF0P2]
[Jd30;Zj^P80];Bd0`2EUID3065QH@80:RXZ0`0;2`/301/K6`80Fe]K0`2YZJT20=CDe0<0`<300`29
RHT2079bLP<0SXj>0P2FUYH3055AD@<0:2PX0P1eMGD30;Jf]P80/;2`0`2FUYH305MGE`80A4A40`25
QHD30;bl_080Ph>30`14A4@209:BTP<0VIVI0`1NGUh2092@T0<0N7Qh0P1oOgl307inOPD0oooo0@00
00010?ooo`003`3oool00`000000oooo0?ooo`020?ooo`D0OWin0`1oOgl20820P0<0MGEe0`2=SHd2
08B4Q0<0Fe]K0P23Ph<30:n_[`<0Q8B40P1;Bd/304i>CP<0YZJV0P2JVYX30<_;b`80QHF50`0;2`/3
04A4A080V9RH0`2>SXh207IfMP<0UIFE0`3>c/h20=cLg0<0Zj^[0P1JFUX301TI6@<01PH60P0M7Ad3
04a<C0<0Ng]k0P2JVYX30:NWY`80Y:BT0`2JVYX308n?S`80QhN70`26QXH208Z:RP<0U9BD0`2QXJ42
0:n_[`<0_Kfm0P39bLT30=CDe0<0gMgM0P3Shn<30>OWi`80j>SX0`3WinL30>?Sh`80gMgM0`3Cdm<3
0<O7a`80^;Rh0`2VYZH209BDU0<0PH610`1bLW8206QXJ0<0IVIV0P1]KFd307inOP<0UYJF0P2][Jd3
0;Zj^P80];Bd0`2EUID3065QH@80:RXZ0`0;2`/301/K6`80Fe]K0`2YZJT20=CDe0<0`<300`29RHT2
079bLP<0SXj>0P2FUYH3055AD@<0:2PX0P1eMGD30;Jf]P80/;2`0`2FUYH305MGE`80A4A40`25QHD3
0;bl_080Ph>30`14A4@209:BTP<0VIVI0`1NGUh2092@T0<0N7Qh0P1oOgl307inOPD0oooo0@000001
0?ooo`003`3oool00`000000oooo0?ooo`020?ooo`D0OWin0`21PH4207UiN@<0OGem0`2BTY8206UY
J@<0IfMW0P2SXj<309nOW`<0GUiN0P16ATH307moO`<0[jn_0P2WYjL30;Rh^080;2`/0`0G5aL308>3
P`80YZJV0`24Q8@208F5Q@<0][Jf0`3EeMD20;Jf]P<0JFUY0P0Q8B4300<30`<05aLG0P18B4P307in
OP<0Y:BT0P2e]KD30;>c/`80YjNW0`2IVIT308n?S`80S8b<0`2@T90209ZJVP<0YZJV0`2d];@20<;2
`P<0c/k>0P3Hf=P30=oOg`<0iNGU0P3YjNT30>c/k0D0kNg]0`3/k>`20>SXj0<0hn?S0`3Jf]X20<o?
c`<0`<300P2_[jl309bLW0<0RHV90P1hN7P306a/K080JFUY0`1aLG4308B4Q080WYjN0`2h^;P20<G5
a@<0^k^k0`2EUID205YJFP<08R8R0`0:2PX202DU9@<0K6a/0P2d];@30<c<c0<0Z:RX0P1lO7`30861
P@80XZ:R0`23Ph<3034a<@80?3`l0`2FUYH20;:b/P<0[:b/0`20P80204A4A0<0F5QH0`2VYZH20;2`
/0<0F5QH0P1MGEd30:j^[P<0JVYZ0P1oOgl30820P0D0OWin1@3oool100000040oooo000?0?ooo`80
00000`3oool507inOP<0PH610P1iNGT307emO@<0TY:B0P1YJFT306MWI`80Xj>S0`2OWil305iNGP80
ATI60`1oOgl30:n_[`80YjNW0`2h^;P202`/;0<05aLG0`23Ph<20:JVYP<0Q8B40P25QHD30;Jf]P<0
eMGE0P2f][H306UYJ@808B4Q0`030`<301LG5`80B4Q80`1nOWh30:BTY080]KFe0`2c/k<20:NWY`<0
VIVI0`2?Shl208b<S0<0T92@0P2JVYX30:JVYP<0];Bd0P32`/830<k>cP80f=SH0`3Ogml30>GUi@80
jNWY0`3/k>`50>g]k@<0k>c/0P3Xj>P30>?Sh`<0f][J0P3?cll30<30`080[jn_0`2LW9`308V9R@80
N7Qh0`1/K6`206UYJ@<0LG5a0`24Q8@209jNWP<0^;Rh0P35aLD30;^k^`<0UIFE0P1JFUX3028R8P<0
2PX:0P0U9BD306a/K080];Bd0`3<c<`30:RXZ080O7al0`21PH420::RXP<0Ph>30`0a<C4203`l?0<0
UYJF0P2b/[830:b/[0<0P8200P14A4@305QHF0<0YZJV0P2`/;0305QHF080GEeM0`2^[Zh306YZJP80
Ogmo0`20P80507inOP@0oooo0P0000010?ooo`003`3oool00`000000oooo0?ooo`020?ooo`D0OWin
0`21PH4207UiN@<0OGem0`2BTY8206UYJ@<0IfMW0P2SXj<309nOW`<0GUiN0P16ATH307moO`<0[jn_
0P2WYjL30;Rh^080;2`/0`0G5aL308>3P`80YZJV0`24Q8@208F5Q@<0][Jf0`3EeMD20;Jf]P<0JFUY
0P0Q8B4300<30`<05aLG0P18B4P307inOP<0Y:BT0P2e]KD30;>c/`80YjNW0`2IVIT308n?S`80S8b<
0`2@T90209ZJVP<0YZJV0`2d];@20<;2`P<0c/k>0P3Hf=P30=oOg`<0iNGU0P3YjNT30>c/k0D0kNg]
0`3/k>`20>SXj0<0hn?S0`3Jf]X20<o?c`<0`<300P2_[jl309bLW0<0RHV90P1hN7P306a/K080JFUY
0`1aLG4308B4Q080WYjN0`2h^;P20<G5a@<0^k^k0`2EUID205YJFP<08R8R0`0:2PX202DU9@<0K6a/
0P2d];@30<c<c0<0Z:RX0P1lO7`30861P@80XZ:R0`23Ph<3034a<@80?3`l0`2FUYH20;:b/P<0[:b/
0`20P80204A4A0<0F5QH0`2VYZH20;2`/0<0F5QH0P1MGEd30:j^[P<0JVYZ0P1oOgl30820P0D0OWin
1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo201oOgl207IfMP<0S8b<0`22
PX8205eMG@<0R8R80P2/[:`307YjNP<0BDU90P1KFe/30;>c/`<0W9bL0P2i^KT306e]K@8010@40`1F
EUH30:j^[P80W9bL0`21PH4209jNWP<0alO70`2m_Kd207YjNP<0;Bd]0P0410@3014A4@<0@d=30P1o
Ogl30:f][@<0`L710P30`<030;:b/P80X:2P0`2CTi<308j>SP80T92@0`2HV9P20:BTY0<0/;2`0`2l
_;`20<K6aP<0c/k>0P3Cdm<30=KFeP<0f=SH0P3IfMT30=[JfP80fm_K0`3Lg=`50=gMg@<0g=cL0`3J
f]X20=GEe@<0c<c<0P30`<030;2`/0<0WIfM0P2:RXX307QhN080K6a/0`1YJFT307=cL`80RHV90`2V
YZH20<;2`P<0c/k>0`30`<0209>CT`<0De=C0`0K6a/200`<30<0=3@d0P1oOgl30;Zj^P<0_;bl0P2A
TI4307YjNP80V9RH0`2WYjL3065QH@8092@T0`1QHF420:FUY@<0/;2`0`2TY:@206=SH`<0@D510`1i
NGT20;Zj^P<0Rh^;0P15ADD308n?S`<0V9RH0P1QHF4308n?S`80NGUi0`1oOgl50?ooo`4000000@3o
ool000l0oooo00<000000?ooo`3oool00P3oool807moO`80MWIf0`2<S8`308:2PP80GEeM0`28R8P2
0:b/[0<0NWYj0`19BDT205]KF`<0/k>c0`2LW9`20;Vi^@<0KFe]0P0410@305IFEP<0[Zj^0P2LW9`3
0861P@80WYjN0`37alL30;fm_@80NWYj0`0];Bd200@410<04A4A0`13@d<207moO`<0[Jf]0`31`L42
0<30`0<0/[:b0P2PX:0309>CT`<0SXj>0P2@T90309RHV080Y:BT0`2`/;030;bl_080a/K60`3>c/h2
0=?Cd`<0e]KF0`3Hf=P20=WIf@<0f][J0P3Kfm/30=cLg0D0gMgM0`3Lg=`30=[JfP80eMGE0`3<c<`2
0<30`0<0/;2`0`2MWId208Z:RP<0N7Qh0P1/K6`306UYJ@<0Lg=c0P29RHT30:JVYP80`/;20`3>c/h3
0<30`080Ti>C0`1CDe<301/K6`8030`<0`0d=3@207moO`<0^[Zj0`2l_;`2096AT@<0NWYj0P2HV9P3
0:NWY`<0HF5Q0P0T92@3065QH@80YJFU0`2`/;030:BTY080Hf=S0`11@D4307UiN@80^[Zj0`2;Rh/2
04E5A@<0Shn?0`2HV9P2065QH@<0Shn?0P1iNGT307moO`D0oooo0@0000010?ooo`003`3oool00`00
0000oooo0?ooo`020?ooo`<0OWin0P21PH4307YjNP80OWin0`2@T90306UYJ@80K6a/0`2UYJD209NG
U`<0EeMG0`1>CTh208b<S0<0]kNg0`2KVi/20:6QX@<07QhN0P0S8b<309fMW@<0]kNg0P2;Rh/308Z:
RP80/[:b0`2m_Kd308^;R`80?Cdm0`092@T200`<30<0?Cdm0`1nOWh20;>c/`<0cLg=0`3<c<`20;^k
^`<0YJFU0P2DU9@308^;R`<0RXZ:0P2ATI4309^KV`80YZJV0`2`/;030;Ng]`80^k^k0`2m_Kd20;bl
_0<0^[Zj0`2h^;P20;Jf]P<0]KFe0P2f][H30;Ng]`<0^[Zj0P2n_[h30<;2`P<0a/K60P38b<P30<O7
a`80`/;20`2i^KT30:b/[080VYZJ0`27QhL207EeM@<0JFUY0`1WIfL207=cL`<0S8b<0P2^[Zh30<_;
b`<0emOG0P33`l<308j>SP<0BDU90P0E5AD3018B4P80ATI60`2@T9030;Vi^@80YjNW0`20P80208J6
QP<0[Zj^0`2HV9P203`l?0<0<C4a0P22PX830:NWY`<0/k>c0P2>SXh304];B`<0Cdm?0P2MWId30;>c
/`80GUiN0`1KFe/30:b/[080JVYZ0`21PH4507inOPD0oooo0@0000010?ooo`003`3oool00`000000
oooo0?ooo`020?ooo`<0OWin0P21PH4307YjNP80OWin0`2@T90306UYJ@80K6a/0`2UYJD209NGU`<0
EeMG0`1>CTh208b<S0<0]kNg0`2KVi/20:6QX@<07QhN0P0S8b<309fMW@<0]kNg0P2;Rh/308Z:RP80
/[:b0`2m_Kd308^;R`80?Cdm0`092@T200`<30<0?Cdm0`1nOWh20;>c/`<0cLg=0`3<c<`20;^k^`<0
YJFU0P2DU9@308^;R`<0RXZ:0P2ATI4309^KV`80YZJV0`2`/;030;Ng]`80^k^k0`2m_Kd20;bl_0<0
^[Zj0`2h^;P20;Jf]P<0]KFe0P2f][H30;Ng]`<0^[Zj0P2n_[h30<;2`P<0a/K60P38b<P30<O7a`80
`/;20`2i^KT30:b/[080VYZJ0`27QhL207EeM@<0JFUY0`1WIfL207=cL`<0S8b<0P2^[Zh30<_;b`<0
emOG0P33`l<308j>SP<0BDU90P0E5AD3018B4P80ATI60`2@T9030;Vi^@80YjNW0`20P80208J6QP<0
[Zj^0`2HV9P203`l?0<0<C4a0P22PX830:NWY`<0/k>c0P2>SXh304];B`<0Cdm?0P2MWId30;>c/`80
GUiN0`1KFe/30:b/[080JVYZ0`21PH4507inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo
0?ooo`020?ooo`<0OWin0P21PH4307YjNP80OWin0`2@T90306UYJ@80K6a/0`2UYJD209NGU`<0EeMG
0`1>CTh208b<S0<0]kNg0`2KVi/20:6QX@<07QhN0P0S8b<309fMW@<0]kNg0P2;Rh/308Z:RP80/[:b
0`2m_Kd308^;R`80?Cdm0`092@T200`<30<0?Cdm0`1nOWh20;>c/`<0cLg=0`3<c<`20;^k^`<0YJFU
0P2DU9@308^;R`<0RXZ:0P2ATI4309^KV`80YZJV0`2`/;030;Ng]`80^k^k0`2m_Kd20;bl_0<0^[Zj
0`2h^;P20;Jf]P<0]KFe0P2f][H30;Ng]`<0^[Zj0P2n_[h30<;2`P<0a/K60P38b<P30<O7a`80`/;2
0`2i^KT30:b/[080VYZJ0`27QhL207EeM@<0JFUY0`1WIfL207=cL`<0S8b<0P2^[Zh30<_;b`<0emOG
0P33`l<308j>SP<0BDU90P0E5AD3018B4P80ATI60`2@T9030;Vi^@80YjNW0`20P80208J6QP<0[Zj^
0`2HV9P203`l?0<0<C4a0P22PX830:NWY`<0/k>c0P2>SXh304];B`<0Cdm?0P2MWId30;>c/`80GUiN
0`1KFe/30:b/[080JVYZ0`21PH4507inOPD0oooo0@0000010?ooo`003`3oool00`000000oooo0?oo
o`020?ooo`<0NWYj0P20P80307MgM`80Rh^;0`20P80305mOG`80S8b<0`2XZ:P2075aL@<0BDU90`1Y
JFT20;bl_0<0VIVI0`2TY:@205YJFP<01PH60P1_Kfl30<G5a@<0Y:BT0P22PX8309^KV`80][Jf0`2I
VIT3055AD@804a<C0`0:2PX203De=@<0NWYj0`2f][H20=KFeP<0emOG0`33`l<20:RXZ0<0TI6A0P24
Q8@30861P@<0QXJ60P2?Shl309RHV080X:2P0`2TY:@30:FUY@80Xj>S0`2NWYh209RHV0<0TY:B0`2<
S8`208R8R0D0QHF50`27QhL308b<S080TY:B0`2JVYX30:>SX`80Zj^[0`2b/[850;Fe]@<0[jn_0P2T
Y:@309BDU080P8200`1^KVh5069RHP<0LG5a0P2?Shl30;Fe]@<0eMGE0P3MgMd30<;2`P<0QXJ60P0o
?cl301<C4`807QhN0`1KFe/309fMW@80/;2`0`2@T90207]kN`<0WIfM0`2h^;P207EeM@<0:2PX0P1A
DE4309:BTP<0Zj^[0P2_[jl3071`L0<0@D510P1^KVh30;Ng]`80TI6A0`17AdL308b<S080UYJF0`1S
Hf<208f=S@<0NWYj1@3oool100000040oooo000?0?ooo`030000003oool0oooo0080oooo0`1jNWX2
0820P0<0MgMg0P2;Rh/30820P0<0GemO0P2<S8`30:RXZ080LG5a0`19BDT306UYJ@80_;bl0`2IVIT3
0:BTY080FUYJ0`061PH206m_K`<0aLG50`2TY:@208:2PP<0Vi^K0P2f][H309VIV@<0DE5A0P0C4a<3
00X:2P80=CDe0`1jNWX30;Jf]P80e]KF0`3GemL30<?3``80Z:RX0`2ATI4208B4Q0<0PH610`26QXH2
08n?S`<0V9RH0P2PX:030:BTY0<0YJFU0P2SXj<309jNWP80V9RH0`2BTY8308b<S080R8R81@25QHD3
08N7Q`<0S8b<0P2BTY8309ZJVP<0Xj>S0P2[Zj/30;:b/PD0]KFe0`2_[jl20:BTY0<0U9BD0P20P803
06i^KPD0HV9R0`1aLG4208n?S`<0]KFe0`3EeMD20=gMg@<0`/;20`26QXH203lo?`<04a<C0P0N7Qh3
05]KF`<0WIfM0P2`/;03092@T080Ng]k0`2MWId30;Rh^080MGEe0`0X:2P2055AD@<0TY:B0`2[Zj/2
0:n_[`<0L71`0`11@D4206i^KP<0]kNg0P2ATI4304M7A`<0S8b<0P2FUYH306=SH`80SHf=0`1jNWX5
0?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3oool307moO`80Ng]k0`1nOWh208j>
SP<0JFUY0`1`L7020:FUY@<0Shn?0P1BDU8305MGE`<0V9RH0P2j^[X608b<S0805aLG0`0f=SH20;Rh
^0<0`/;20`2;Rh/208J6QP<0YZJV0P2QXJ4306IVIP<08R8R0P0;2`/302h^;P80Lg=c0`2f][H30=gM
g@80h>3P0`3:b/X30:ZZZP80SXj>0`1lO7`207IfMP<0NGUi0`21PH4208V9R@<0T92@0P2BTY83092@
T0<0RXZ:0P22PX8307UiN@80Kfm_0`1UIFD305eMG@80EeMG0`1DE5@205=CD`<0EEEE0`1JFUX2069R
HP<0K6a/0`1gMgL208B4Q0<0TI6A0P2LW9`30:BTY0<0Z:RX0P2UYJD309^KV`80S8b<0`1iNGT306IV
IP80FUYJ0`1LG5`206i^KP<0TI6A0`2l_;`20=gMg@<0h^;R0`2o_kl207alO0<0=SHf0P0E5AD302h^
;P<0L71`0P2SXj<309nOW`80OWin0`25QHD30;Jf]P80[:b/0`1=CDd202h^;P<0Kfm_0`2IVIT20;Bd
]0<0W9bL0`1DE5@204U9B@<0U9BD0P2f][H306=SH`<0FUYJ0P2ZZZX306][J`80PX:20`1oOgl50?oo
o`4000000@3oool000l0oooo00<000000?ooo`3oool00P3oool307moO`80Ng]k0`1nOWh208j>SP<0
JFUY0`1`L7020:FUY@<0Shn?0P1BDU8305MGE`<0V9RH0P2j^[X608b<S0805aLG0`0f=SH20;Rh^0<0
`/;20`2;Rh/208J6QP<0YZJV0P2QXJ4306IVIP<08R8R0P0;2`/302h^;P80Lg=c0`2f][H30=gMg@80
h>3P0`3:b/X30:ZZZP80SXj>0`1lO7`207IfMP<0NGUi0`21PH4208V9R@<0T92@0P2BTY83092@T0<0
RXZ:0P22PX8307UiN@80Kfm_0`1UIFD305eMG@80EeMG0`1DE5@205=CD`<0EEEE0`1JFUX2069RHP<0
K6a/0`1gMgL208B4Q0<0TI6A0P2LW9`30:BTY0<0Z:RX0P2UYJD309^KV`80S8b<0`1iNGT306IVIP80
FUYJ0`1LG5`206i^KP<0TI6A0`2l_;`20=gMg@<0h^;R0`2o_kl207alO0<0=SHf0P0E5AD302h^;P<0
L71`0P2SXj<309nOW`80OWin0`25QHD30;Jf]P80[:b/0`1=CDd202h^;P<0Kfm_0`2IVIT20;Bd]0<0
W9bL0`1DE5@204U9B@<0U9BD0P2f][H306=SH`<0FUYJ0P2ZZZX306][J`80PX:20`1oOgl50?ooo`40
00000@3oool000l0oooo00<000000?ooo`3oool00P3oool307moO`80Ng]k0`1nOWh208j>SP<0JFUY
0`1`L7020:FUY@<0Shn?0P1BDU8305MGE`<0V9RH0P2j^[X608b<S0805aLG0`0f=SH20;Rh^0<0`/;2
0`2;Rh/208J6QP<0YZJV0P2QXJ4306IVIP<08R8R0P0;2`/302h^;P80Lg=c0`2f][H30=gMg@80h>3P
0`3:b/X30:ZZZP80SXj>0`1lO7`207IfMP<0NGUi0`21PH4208V9R@<0T92@0P2BTY83092@T0<0RXZ:
0P22PX8307UiN@80Kfm_0`1UIFD305eMG@80EeMG0`1DE5@205=CD`<0EEEE0`1JFUX2069RHP<0K6a/
0`1gMgL208B4Q0<0TI6A0P2LW9`30:BTY0<0Z:RX0P2UYJD309^KV`80S8b<0`1iNGT306IVIP80FUYJ
0`1LG5`206i^KP<0TI6A0`2l_;`20=gMg@<0h^;R0`2o_kl207alO0<0=SHf0P0E5AD302h^;P<0L71`
0P2SXj<309nOW`80OWin0`25QHD30;Jf]P80[:b/0`1=CDd202h^;P<0Kfm_0`2IVIT20;Bd]0<0W9bL
0`1DE5@204U9B@<0U9BD0P2f][H306=SH`<0FUYJ0P2ZZZX306][J`80PX:20`1oOgl50?ooo`400000
0@3oool000l0oooo0P00003o0?ooo`d0oooo0P0000010?ooo`003`3oool00`000000oooo0?ooo`3o
0?ooo`d0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`0L0?ooo`030000003oool0oooo
0340oooo00<000000?ooo`3oool0<P3oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo
0340oooo00<000000?ooo`3oool0703oool100000040oooo000?0?ooo`030000003oool0oooo0080
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0080
oooo0@0000010?ooo`003`3ooooo0000014000000@3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-57.8976, -3.36627, 0.383087, \
0.175527}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  DensityGraphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[9]="]
}, Open  ]],

Cell[TextData[{
  "The",
  StyleBox[" ", "MR"],
  StyleBox[ButtonBox["\"DiscretizedMonitorVariables\"",
    ButtonData:>{"PDE.nb", "b:7.1.5"},
    ButtonStyle->"Hyperlink"], "MR"],
  StyleBox[" ", "MR"],
  "option affects the way the event is interpreted for PDEs; with the setting \
True, u[t,x] is replaced by a vector of discretized values. This is much more \
efficient because it avoids explicitly constructing the ",
  StyleBox["InterpolatingFunction", "MR"],
  " to evaluate the event."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Timing[
      sgsol\  = \ 
        First[NDSolve[{D[u[t, x], t, t]\  \[Equal] \ 
                D[u[t, x], x, x]\  - \ Sin[u[t, x]], \ 
              u[0, x]\  \[Equal] \ \[ExponentialE]\^\(-\((x - 5)\)\^2\) + \
\[ExponentialE]\^\(\(-\((x + 5)\)\^2\)/2\), \ \(\(Derivative[1, 0]\)[u]\)[0, 
                  x]\  \[Equal] \ 0, \ 
              u[t, \(-50\)]\  \[Equal] \ u[t, 50]}, \ 
            u, \ {t, 0, 1000}, \ {x, \(-50\), 50}, \ 
            Method \[Rule] {"\<MethodOfLines\>", \
\[IndentingNewLine]"\<DiscretizedMonitorVariables\>" \[Rule] 
                  True, \ \[IndentingNewLine]"\<SpatialDiscretization\>" \
\[Rule] {"\<TensorProductGrid\>", \ "\<DifferenceOrder\>" -> \
"\<Pseudospectral\>"}, 
                Method \[Rule] {"\<EventLocator\>", \ "\<Event\>" \
\[RuleDelayed] \ Abs[First[u[t, x]]]\  - \ 
                        0.01, \ "\<EventLocationMethod\>" -> \
"\<StepBegin\>"}}]]]\)], "Input",
  CellLabel->"In[10]:=",
  CellTags->"ELPDEExample"],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{\(0.29000000000000004`\ Second\), ",", 
        RowBox[{"{", 
          RowBox[{"u", "\[Rule]", 
            
            TagBox[\(InterpolatingFunction[{{0.`, 
                    45.81888712328008`}, {"...", \(-50.`\), 50.`, "..."}}, 
                "<>"]\),
              False,
              Editable->False]}], "}"}]}], "}"}]], "Output",
  CellLabel->"Out[10]=",
  CellTags->"ELPDEExample"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Performance Comparison", "Subsection"],

Cell["\<\
The following example constructs a table making a comparison for \
two different integration methods.\
\>", "Text"],

Cell["\<\
This defines a function which returns the time it takes to compute \
a solution of a mildly damped pendulum equation up to the point at which the \
bob has momentarily been at rest 1000 times.\
\>", "MathCaption"],

Cell[BoxData[
    \(\(EventLocatorTiming[locmethod_, odemethod_]\  := \ 
        Block[{Second\  = \ 1, \ x, \ t, \ 
            p\  = \ 0}, \[IndentingNewLine]First[
            Timing[NDSolve[{\(x''\)[t]\  + \ 1/1000\ \(x'\)[t]\  + \ 
                      Sin[x[t]]\  \[Equal] \ 0, \ 
                  x[0]\  \[Equal] \ 3, \ \(x'\)[0]\  \[Equal] \ 0}, \ 
                x, \ {t, \[Infinity]}, \ 
                Method \[Rule] {"\<EventLocator\>", \ "\<Event\>" \[Rule] \
\(x'\)[t], \ "\<EventAction\>" \[RuleDelayed] \ 
                      If[\(p++\)\  \[GreaterEqual] \ 1000, \ 
                        Throw[end\  = \ 
                            t, \ "\<StopIntegration\>"]], \
"\<EventLocationMethod\>" \[Rule] locmethod, \ "\<Method\>" \[Rule] 
                      odemethod}, \[IndentingNewLine]MaxSteps \[Rule] \
\[Infinity]]]]\[IndentingNewLine]];\)\)], "Input",
  CellLabel->"In[36]:="],

Cell["\<\
This uses the function to make a table comparing the different \
location methods for two different ODE integration methods.\
\>", \
"MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(elmethods\  = \ {"\<StepBegin\>", "\<StepEnd\>", \ \
"\<LinearInterpolation\>", \ {"\<Brent\>", "\<SolutionApproximation\>" -> \
"\<CubicHermiteInterpolation\>"}, \ Automatic};\)\), "\[IndentingNewLine]", 
    \(\(odemethods\  = \ {Automatic, \ "\<ExplicitRungeKutta\>"};\)\), "\
\[IndentingNewLine]", 
    \(TableForm[Outer[EventLocatorTiming, \ elmethods, \ odemethods, 1], \ 
      TableHeadings \[Rule] {elmethods, \ odemethods}]\)}], "Input",
  CellLabel->"In[37]:="],

Cell[BoxData[
    TagBox[
      TagBox[GridBox[{
            {
              StyleBox["\[Null]",
                ShowStringCharacters->False], 
              TagBox["Automatic",
                HoldForm], 
              TagBox["\<\"ExplicitRungeKutta\"\>",
                HoldForm]},
            {
              TagBox["\<\"StepBegin\"\>",
                HoldForm], "0.33000000000000085`", "0.30000000000000077`"},
            {
              TagBox["\<\"StepEnd\"\>",
                HoldForm], "0.33000000000000007`", "0.33`"},
            {
              TagBox["\<\"LinearInterpolation\"\>",
                HoldForm], "0.36999999999999994`", "0.32000000000000073`"},
            {
              
              TagBox[\({"Brent", 
                  "SolutionApproximation" \[Rule] 
                    "CubicHermiteInterpolation"}\),
                HoldForm], "0.5200000000000005`", "0.4900000000000009`"},
            {
              TagBox["Automatic",
                HoldForm], "0.5799999999999988`", "0.6000000000000005`"}
            },
          RowSpacings->1,
          ColumnSpacings->1,
          ColumnAlignments->{Left},
          RowLines->{True, False},
          ColumnLines->{True, False}],
        {OutputFormsDump`HeadedRows, OutputFormsDump`HeadedColumns}],
      Function[ BoxForm`e$, 
        MatrixForm[ 
        BoxForm`e$, TableHeadings -> {{"StepBegin", "StepEnd", 
          "LinearInterpolation", {"Brent", "SolutionApproximation" -> 
          "CubicHermiteInterpolation"}, Automatic}, {Automatic, 
          "ExplicitRungeKutta"}}]]]], "Output",
  CellLabel->"Out[39]//TableForm="]
}, Open  ]],

Cell["\<\
While simple step begin/end and linear interpolation location are \
essentially the same low cost,  the better location methods are more \
expensive.  The default location method is particularly expensive for the \
explicit Runge-Kutta method because that does not yet support a continuous \
output formula - it therefore needs to repeatedly invoke the method with \
different step sizes during the local minimization.\
\>", "Text"],

Cell["\<\
It is worth noting that, often, a significant part of the extra \
time for computing events arises from the need to evaluate the event \
functions at each time step to check for the possibility of a sign \
change.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(TableForm[\[IndentingNewLine]{Map[\[IndentingNewLine]Block[{Second = 1, 
                x, t, p = 0}, \[IndentingNewLine]First[
                Timing[NDSolve[{\(x''\)[t] + 1/1000\ \(x'\)[t] + 
                          Sin[x[t]] \[Equal] 0, 
                      x[0] \[Equal] 3, \(x'\)[0] \[Equal] 0}, x, {t, end}, 
                    Method \[Rule] #, 
                    MaxSteps \[Rule] \[Infinity]]]]] &, \
\[IndentingNewLine]odemethods\[IndentingNewLine]]}, \
\[IndentingNewLine]TableHeadings \[Rule] {None, odemethods}]\)], "Input",
  CellLabel->"In[40]:="],

Cell[BoxData[
    TagBox[
      TagBox[GridBox[{
            {
              TagBox["Automatic",
                HoldForm], 
              TagBox["\<\"ExplicitRungeKutta\"\>",
                HoldForm]},
            {"0.21999999999999914`", "0.2300000000000003`"}
            },
          RowSpacings->1,
          ColumnSpacings->1,
          ColumnAlignments->{Left},
          RowLines->{True, False},
          ColumnLines->False],
        {None, OutputFormsDump`HeadedColumns}],
      Function[ BoxForm`e$, 
        MatrixForm[ 
        BoxForm`e$, TableHeadings -> {None, {Automatic, 
          "ExplicitRungeKutta"}}]]]], "Output",
  CellLabel->"Out[40]//TableForm="]
}, Open  ]],

Cell["\<\
An optimization is performed for event functions involving only the \
independent variable. Such events are detected automatically at \
initialization time. For example, this has the advantage that interpolation \
of the solution of the dependent variables is not carried out during at each \
step if the local optimization search - it is deferred until the value of the \
independent variable has been found.\
\>", "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Limitations", "Section",
  CellTags->"s:4"],

Cell[TextData[{
  "One limitation of the event locator method is that since the event \
function is only checked for sign changes over a step interval, if the event \
function has multiple roots in a step interval, all or some of the events may \
be missed.  This typically only happens when the solution to the ODE varies \
much more slowly than the event function.  When you suspect that this may \
have occurred, the simplest solution is to decrease the maximum step size the \
method can take by using the ",
  StyleBox["MaxStepSize", "MR"],
  " option to ",
  StyleBox["NDSolve", "MR"],
  ".  More sophisticated approaches can be taken, but the best approach \
depends on what is being computed.   An example is shown below which \
demonstrates the problem and shows two approaches for fixing it."
}], "Text"],

Cell[TextData[{
  "This should compute the number of positive integers less than ",
  Cell[BoxData[
      \(TraditionalForm\`\[ExponentialE]\^5\)]],
  "(there are 148).  However, most are missed because the method is taking \
large time steps because the solution x[t] is so simple."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Block[{n\  = \ 0}, 
      NDSolve[{\(x'\)[t]\  \[Equal] \ x[t], \ 
          x[\(-1\)]\  \[Equal] \ \[ExponentialE]\^\(-1\)}, \ x, \ {t, 5}, 
        Method \[Rule] {"\<EventLocator\>", \ "\<Event\>" \[Rule] \ 
              Sin[\[Pi]\ x[t]], \ "\<EventAction\>" \[RuleDelayed] \(n++\)}]; 
      n]\)], "Input",
  CellLabel->"In[17]:="],

Cell[BoxData[
    \(18\)], "Output",
  CellLabel->"Out[17]="]
}, Open  ]],

Cell["\<\
This restricts the maximum step size so that all the events are \
found.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Block[{n\  = \ 0}, 
      NDSolve[{\(x'\)[t]\  \[Equal] \ x[t], \ 
          x[\(-1\)]\  \[Equal] \ \[ExponentialE]\^\(-1\)}, \ x, \ {t, 5}, 
        Method \[Rule] {"\<EventLocator\>", \ "\<Event\>" \[Rule] \ 
              Sin[\[Pi]\ x[
                    t]], \ "\<EventAction\>" \[RuleDelayed] \(n++\)}, \ 
        MaxStepSize \[Rule] 0.001]; n]\)], "Input",
  CellLabel->"In[18]:="],

Cell[BoxData[
    \(148\)], "Output",
  CellLabel->"Out[18]="]
}, Open  ]],

Cell["\<\
It is quite apparent from the nature of the example problem that if \
the endpoint is increased, it is likely that a smaller maximum step size may \
be required.  Taking very small steps everywhere is quite inefficient.  It is \
possible to introduce an adaptive time step restriction be setting up a \
variable which varies on the same time scale  as the event function.\
\>", \
"Text"],

Cell["\<\
This introduces an additional function to integrate which is the \
event function.  With this modification and allowing the method to take as \
many steps is needed, it is possible to find the correct value up to t = 10 \
in a reasonable amount of time.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Block[{n\  = \ 0}, 
      NDSolve[{\(x'\)[t]\  \[Equal] \ x[t], \ 
          x[\(-1\)]\  \[Equal] \ \[ExponentialE]\^\(-1\), \ \(y'\)[
              t]\  \[Equal] \ D[Sin[\[Pi]\ x[t]], t], \ 
          y[\(-1\)]\  \[Equal] \ Sin[\[Pi]\ \[ExponentialE]\^\(-1\)]}, \ {x, 
          y}, \ {t, 10}, 
        Method \[Rule] {"\<EventLocator\>", \ "\<Event\>" \[Rule] \ 
              y[t], \ "\<EventAction\>" \[RuleDelayed] \(n++\)}, \ 
        MaxSteps \[Rule] \[Infinity]]; n]\)], "Input",
  CellLabel->"In[19]:="],

Cell[BoxData[
    \(22026\)], "Output",
  CellLabel->"Out[19]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
  CellTags->"s:5"],

Cell[CellGroupData[{

Cell["EventLocator options", "Subsection",
  CellTags->"EventLocatorOptionSummary"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {Cell[TextData[StyleBox["\"Direction\"", "MR"]]], 
            StyleBox["All",
              
              "MR"], \(The\ direction\ of\ zero\ crossing\ to\ allow\ for\ \
the\ event . 
                1\ means\ from\ negative\ to\ positive, \(-1\)\ means\ from\ \
positive\ to\ negative, and\ All\ includes\ both\ \(\(directions\)\(.\)\)\)},
          {Cell[TextData[StyleBox["\"Event\"", "MR"]]], 
            StyleBox["None",
              
              "MR"], \(An\ expression\ that\ defines\ the\ event . 
                An\ event\ occurs\ at\ points\ where\ substituting\ the\ \
numerical\ values\ of\ the\ problem\ variables\ makes\ the\ expression\ equal\
\ to\ \(\(zero\)\(.\)\)\)},
          {Cell[TextData[StyleBox["\"EventAction\"", "MR"]]], Cell[TextData[
            StyleBox["Throw[Null, \"StopIntegration\"]", "MR"]]], 
            Cell["\<\
What to do when an event occurs.   Problem variables \
are substituted with their numerical values at the event.  In general, you \
need to use RuleDelayed (\[RuleDelayed])  to prevent it from being evaluated \
except with numerical values.\
\>", "Text"]},
          {Cell[TextData[StyleBox["\"EventLocationMethod\"", "MR"]]], 
            StyleBox["Automatic",
              "MR"], 
            Cell["\<\
The method to use for refining the location of a given \
event.  \
\>"]},
          {Cell[TextData[StyleBox["\"Method\"", "MR"]]], 
            StyleBox["Automatic",
              "MR"], Cell[
            "The method to use for integrating the system of ODEs.  "]}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell["EventLocator method options", "Caption"]
}, Open  ]],

Cell[CellGroupData[{

Cell["EventLocationMethod options", "Subsection",
  CellTags->"EventLocationMethodOptionSummary"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["\"Brent\"", "MR"]]], 
            RowBox[{Cell[TextData[{
                "Use ",
                StyleBox["FindRoot", "MR"],
                " with the ",
                StyleBox["Method->", "MR"],
                StyleBox[Cell[BoxData[
                    ButtonBox["Brent",
                      ButtonFunction:>(FrontEndExecute[ {
                          FrontEnd`NotebookLocate[ #2], 
                          FrontEnd`HelpBrowserLookup[ 
                          "RefGuide", 
                            "Advanced Documentation: Unconstrained \
Optimization"]}]&),
                      ButtonData:>{
                        
                        FrontEnd`FileName[ {$TopDirectory, "Documentation", 
                          "English", "RefGuide", "AdvancedDocumentation", 
                          "Optimization"}, "Unconstrained.nb"], "t:13"},
                      Active->True,
                      ButtonStyle->"RefGuideLinkText",
                      ButtonNote->"Advanced Documentation"]], "MR"], "MR"],
                " to locate the event."
              }], "Text"], "  ", "This", " ", "is", " ", "the", " ", 
              "default", " ", "with", " ", "the", " ", "setting", " ", 
              RowBox[{
                StyleBox["Automatic",
                  "MR"], "."}]}]},
          {Cell[TextData[StyleBox["\"LinearInterpolation\"", "MR"]]], 
            Cell["\<\
Locate the event time using linear interpolation. Cubic \
Hermite interpolation is then used to find the solution at the event \
time.\
\>"]},
          {Cell[TextData[StyleBox[
            "\"StepBegin\"", 
              "MR"]]], \(The\ event\ is\ given\ by\ the\ solution\ at\ the\ \
beginning\ of\ the\ \(\(step\)\(.\)\)\)},
          {Cell[TextData[StyleBox[
            "\"StepEnd\"", 
              "MR"]]], \(The\ event\ is\ given\ by\ the\ solution\ at\ the\ \
end\ of\ the\ \(\(step\)\(.\)\)\)}
          }], TextForm]], "DefinitionBox",
  GridBoxOptions->{ColumnWidths->{0.4, 0.6}}],

Cell["Settings for the EventLocationMethod option.", "Caption"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Brent options", "Subsection",
  CellTags->"BrentOptionSummary"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {Cell[TextData[StyleBox["\"MaxIterations\"", "MR"]]], 
            StyleBox["100",
              
              "MR"], \(The\ maximum\ number\ of\ iterations\ to\ use\ for\ \
locating\ an\ event\ within\ a\ step\ of\ the\ \(\(method\)\(.\)\(\ \ \
\)\)\)},
          {Cell[TextData[StyleBox["\"AccuracyGoal\"", "MR"]]], 
            StyleBox["Automatic",
              "MR"], 
            RowBox[{
              RowBox[{
              "Accuracy", " ", "goal", " ", "setting", " ", "passed", " ", 
                "to", " ", 
                RowBox[{
                  StyleBox["FindRoot",
                    "MR"], ".", " ", "If"}], " ", 
                StyleBox["Automatic",
                  "MR"]}], ",", " ", 
              RowBox[{"the", " ", "value", " ", "passed", " ", "to", " ", 
                StyleBox["FindRoot",
                  "MR"], " ", "is", " ", "based", " ", "on", " ", "the", " ", 
                "local", " ", "error", " ", "setting", " ", "for", " ", 
                RowBox[{
                  StyleBox["NDSolve",
                    "MR"], "."}]}]}]},
          {Cell[TextData[StyleBox["\"PrecisionGoal\"", "MR"]]], 
            StyleBox["Automatic",
              "MR"], 
            RowBox[{
              RowBox[{
              "Precision", " ", "goal", " ", "setting", " ", "passed", " ", 
                "to", " ", 
                RowBox[{
                  StyleBox["FindRoot",
                    "MR"], ".", " ", "If"}], " ", 
                StyleBox["Automatic",
                  "MR"]}], ",", " ", 
              RowBox[{"the", " ", "value", " ", "passed", " ", "to", " ", 
                StyleBox["FindRoot",
                  "MR"], " ", "is", " ", "based", " ", "on", " ", "the", " ", 
                "local", " ", "error", " ", "setting", " ", "for", " ", 
                RowBox[{
                  StyleBox["NDSolve",
                    "MR"], "."}]}]}]},
          {Cell[TextData[StyleBox["\"SolutionApproximation\"", "MR"]]], 
            StyleBox["Automatic",
              "MR"], 
            RowBox[{
              RowBox[{
              "How", " ", "to", " ", "approximate", " ", "the", " ", 
                "solution", " ", "for", " ", "evaluating", " ", "the", " ", 
                "event", " ", "function", " ", "during", " ", "the", " ", 
                "refinement", " ", \(process . \ \ Can\), " ", "be", " ", 
                StyleBox["Automatic",
                  "MR"], " ", "or", " ", Cell[TextData[StyleBox[
                "CubicHermiteInterpolation", "MR"]]]}], "."}]}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell["Options for event location method Brent.", "Caption"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.1 for Macintosh",
ScreenRectangle->{{33, 1344}, {0, 986}},
AutoGeneratedPackage->None,
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{650, 500},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"EventLocator",
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
Magnification->1,
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "s:1"->{
    Cell[1832, 57, 50, 1, 61, "Section",
      CellTags->"s:1"]},
  "s:2"->{
    Cell[34245, 991, 60, 1, 61, "Section",
      CellTags->"s:2"]},
  "s:3"->{
    Cell[81099, 2400, 46, 1, 61, "Section",
      CellTags->"s:3"]},
  "ELPDEExample"->{
    Cell[705717, 23601, 996, 20, 188, "Input",
      CellTags->"ELPDEExample"],
    Cell[706716, 23623, 442, 12, 41, "Output",
      CellTags->"ELPDEExample"]},
  "s:4"->{
    Cell[713299, 23800, 49, 1, 61, "Section",
      CellTags->"s:4"]},
  "s:5"->{
    Cell[716903, 23903, 52, 1, 61, "Section",
      CellTags->"s:5"]},
  "EventLocatorOptionSummary"->{
    Cell[716980, 23908, 83, 1, 36, "Subsection",
      CellTags->"EventLocatorOptionSummary"]},
  "EventLocationMethodOptionSummary"->{
    Cell[718948, 23956, 97, 1, 36, "Subsection",
      CellTags->"EventLocationMethodOptionSummary"]},
  "BrentOptionSummary"->{
    Cell[721214, 24011, 69, 1, 36, "Subsection",
      CellTags->"BrentOptionSummary"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"s:1", 724997, 24105},
  {"s:2", 725074, 24108},
  {"s:3", 725153, 24111},
  {"ELPDEExample", 725242, 24114},
  {"s:4", 725414, 24119},
  {"s:5", 725496, 24122},
  {"EventLocatorOptionSummary", 725600, 24125},
  {"EventLocationMethodOptionSummary", 725736, 24128},
  {"BrentOptionSummary", 725865, 24131}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 31, 0, 88, "Chapter"],

Cell[CellGroupData[{
Cell[1832, 57, 50, 1, 61, "Section",
  CellTags->"s:1"],
Cell[1885, 60, 266, 5, 36, "Text"],
Cell[2154, 67, 510, 13, 51, "Text"],
Cell[2667, 82, 355, 8, 51, "Text"],
Cell[3025, 92, 322, 6, 51, "Text"],
Cell[3350, 100, 173, 4, 56, "MathCaption"],
Cell[3526, 106, 463, 10, 90, "Input",
  InitializationCell->True],
Cell[3992, 118, 208, 4, 36, "Text"],
Cell[4203, 124, 137, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[4365, 131, 286, 7, 42, "Input"],
Cell[4654, 140, 290, 9, 25, "Output"]
}, Open  ]],
Cell[4959, 152, 311, 7, 38, "Text"],
Cell[5273, 161, 170, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[5468, 169, 321, 6, 58, "Input"],
Cell[5792, 177, 17006, 481, 185, 4584, 322, "GraphicsData", "PostScript", \
"Graphics"],
Cell[22801, 660, 154, 4, 25, "Output"]
}, Open  ]],
Cell[22970, 667, 237, 6, 37, "Text"],
Cell[23210, 675, 269, 5, 36, "Text"],
Cell[23482, 682, 167, 4, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[23674, 690, 442, 8, 58, "Input"],
Cell[24119, 700, 395, 9, 21, "Print"],
Cell[24517, 711, 391, 9, 21, "Print"],
Cell[24911, 722, 398, 9, 21, "Print"],
Cell[25312, 733, 249, 7, 25, "Output"]
}, Open  ]],
Cell[25576, 743, 267, 8, 37, "Text"],
Cell[25846, 753, 385, 7, 51, "Text"],
Cell[26234, 762, 865, 24, 104, "MathCaption"],

Cell[CellGroupData[{
Cell[27124, 790, 466, 9, 74, "Input"],
Cell[27593, 801, 1031, 22, 115, "Output"]
}, Open  ]],
Cell[28639, 826, 760, 16, 94, "Text"],
Cell[29402, 844, 438, 7, 64, "Text"],
Cell[29843, 853, 579, 21, 53, "Text"],
Cell[30425, 876, 179, 3, 56, "MathCaption"],
Cell[30607, 881, 363, 7, 90, "Input"],
Cell[30973, 890, 135, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[31133, 897, 336, 6, 42, "Input"],
Cell[31472, 905, 290, 9, 25, "Output"]
}, Open  ]],
Cell[31777, 917, 321, 10, 38, "Text"],
Cell[32101, 929, 352, 6, 50, "Text"],
Cell[32456, 937, 226, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[32707, 945, 598, 10, 122, "Input"],
Cell[33308, 957, 384, 12, 25, "Output"]
}, Open  ]],
Cell[33707, 972, 299, 5, 50, "Text"],
Cell[34009, 979, 199, 7, 23, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[34245, 991, 60, 1, 61, "Section",
  CellTags->"s:2"],
Cell[34308, 994, 544, 10, 65, "Text"],
Cell[34855, 1006, 390, 7, 50, "Text"],
Cell[35248, 1015, 281, 7, 37, "Text"],
Cell[35532, 1024, 280, 5, 50, "Text"],

Cell[CellGroupData[{
Cell[35837, 1033, 51, 0, 36, "Subsection"],
Cell[35891, 1035, 831, 14, 93, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[36759, 1054, 48, 0, 36, "Subsection"],
Cell[36810, 1056, 455, 8, 66, "Text"],
Cell[37268, 1066, 144, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[37437, 1073, 843, 17, 170, "Input"],
Cell[38283, 1092, 21617, 688, 184, 7427, 508, "GraphicsData", "PostScript", \
"Graphics"],
Cell[59903, 1782, 155, 4, 25, "Output"]
}, Open  ]],
Cell[60073, 1789, 225, 4, 36, "Text"],
Cell[60301, 1795, 64, 0, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[60390, 1799, 171, 4, 42, "Input"],
Cell[60564, 1805, 15341, 452, 184, 4429, 312, "GraphicsData", "PostScript", \
"Graphics"],
Cell[75908, 2259, 155, 4, 25, "Output"]
}, Open  ]],
Cell[76078, 2266, 361, 6, 50, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[76476, 2277, 36, 0, 36, "Subsection"],
Cell[76515, 2279, 1385, 33, 81, "Text"],
Cell[77903, 2314, 809, 12, 107, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[78749, 2331, 32, 0, 36, "Subsection"],
Cell[78784, 2333, 169, 4, 36, "Text"],
Cell[78956, 2339, 70, 0, 40, "MathCaption"],
Cell[79029, 2341, 129, 2, 42, "Input"],
Cell[79161, 2345, 170, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[79356, 2353, 689, 12, 170, "Input"],
Cell[80048, 2367, 242, 5, 21, "Print"],
Cell[80293, 2374, 237, 5, 21, "Print"],
Cell[80533, 2381, 263, 5, 21, "Print"],
Cell[80799, 2388, 239, 5, 21, "Print"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[81099, 2400, 46, 1, 61, "Section",
  CellTags->"s:3"],

Cell[CellGroupData[{
Cell[81170, 2405, 34, 0, 36, "Subsection"],
Cell[81207, 2407, 230, 7, 24, "Text"],
Cell[81440, 2416, 130, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[81595, 2423, 417, 8, 74, "Input"],
Cell[82015, 2433, 195, 5, 25, "Output"]
}, Open  ]],
Cell[82225, 2441, 136, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[82386, 2448, 498, 10, 122, "Input"],
Cell[82887, 2460, 18211, 684, 184, 6792, 538, "GraphicsData", "PostScript", \
"Graphics"],
Cell[101101, 3146, 155, 4, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[101305, 3156, 58, 0, 36, "Subsection"],
Cell[101366, 3158, 402, 7, 51, "Text"],
Cell[101771, 3167, 266, 6, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[102062, 3177, 722, 15, 76, "Input"],
Cell[102787, 3194, 538, 16, 41, "Output"]
}, Open  ]],
Cell[103340, 3213, 89, 3, 22, "Text"],
Cell[103432, 3218, 196, 5, 37, "Text"],
Cell[103631, 3225, 105, 3, 40, "MathCaption"],
Cell[103739, 3230, 1401, 26, 134, "Input"],
Cell[105143, 3258, 93, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[105261, 3265, 68, 2, 26, "Input"],
Cell[105332, 3269, 77, 2, 25, "Output"]
}, Open  ]],
Cell[105424, 3274, 112, 3, 22, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[105573, 3282, 47, 0, 36, "Subsection"],

Cell[CellGroupData[{
Cell[105645, 3286, 56, 0, 34, "Subsubsection"],
Cell[105704, 3288, 124, 3, 22, "Text"],
Cell[105831, 3293, 105, 3, 22, "Text"],
Cell[105939, 3298, 154, 6, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[106118, 3308, 322, 7, 74, "Input"],
Cell[106443, 3317, 1183, 29, 84, "Output"]
}, Open  ]],
Cell[107641, 3349, 292, 9, 36, "Text"],
Cell[107936, 3360, 754, 17, 82, "Text"],
Cell[108693, 3379, 421, 7, 64, "Text"],
Cell[109117, 3388, 74, 0, 40, "MathCaption"],
Cell[109194, 3390, 82, 2, 26, "Input"],
Cell[109279, 3394, 315, 8, 56, "MathCaption"],
Cell[109597, 3404, 740, 14, 170, "Input"],
Cell[110340, 3420, 229, 5, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[110594, 3429, 205, 4, 42, "Input"],
Cell[110802, 3435, 85216, 3796, 294, 63177, 3519, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[196033, 7234, 155, 3, 22, "Text"],
Cell[196191, 7239, 329, 8, 56, "MathCaption"],
Cell[196523, 7249, 774, 14, 186, "Input"],
Cell[197300, 7265, 229, 5, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[197554, 7274, 206, 4, 42, "Input"],
Cell[197763, 7280, 82112, 3755, 294, 63118, 3515, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[279924, 11041, 37, 0, 34, "Subsubsection"],
Cell[279964, 11043, 202, 4, 36, "Text"],
Cell[280169, 11049, 372, 8, 74, "Input"],
Cell[280544, 11059, 137, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[280706, 11066, 96, 2, 26, "Input"],
Cell[280805, 11070, 692, 19, 74, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[281534, 11094, 101, 2, 26, "Input"],
Cell[281638, 11098, 641, 18, 25, "Output"]
}, Open  ]],
Cell[282294, 11119, 65, 0, 40, "MathCaption"],
Cell[282362, 11121, 422, 8, 74, "Input"],
Cell[282787, 11131, 126, 3, 40, "MathCaption"],
Cell[282916, 11136, 577, 11, 122, "Input"],
Cell[283496, 11149, 133, 3, 40, "MathCaption"],
Cell[283632, 11154, 900, 18, 218, "Input"],
Cell[284535, 11174, 173, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[284733, 11182, 1327, 25, 298, "Input"],
Cell[286063, 11209, 83730, 3751, 184, 64536, 3509, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[369854, 14967, 35, 0, 36, "Subsection"],
Cell[369892, 14969, 408, 9, 53, "Text"],
Cell[370303, 14980, 194, 4, 56, "MathCaption"],
Cell[370500, 14986, 1494, 26, 250, "Input"],
Cell[371997, 15014, 233, 5, 56, "MathCaption"],
Cell[372233, 15021, 930, 17, 234, "Input"],
Cell[373166, 15040, 157, 3, 56, "MathCaption"],
Cell[373326, 15045, 1744, 29, 362, "Input"],
Cell[375073, 15076, 194, 8, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[375292, 15088, 189, 4, 42, "Input"],
Cell[375484, 15094, 20104, 737, 210, 7927, 582, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[395603, 15834, 69, 0, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[395697, 15838, 205, 4, 42, "Input"],
Cell[395905, 15844, 22646, 802, 228, 8650, 624, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[418566, 16649, 63, 0, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[418654, 16653, 242, 5, 42, "Input"],
Cell[418899, 16660, 25390, 1016, 200, 11924, 845, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[444338, 17682, 37, 0, 36, "Subsection"],
Cell[444378, 17684, 485, 12, 52, "Text"],
Cell[444866, 17698, 224, 5, 56, "MathCaption"],
Cell[445093, 17705, 1309, 24, 317, "Input"],
Cell[446405, 17731, 179, 4, 56, "MathCaption"],
Cell[446587, 17737, 139, 3, 26, "Input"],
Cell[446729, 17742, 669, 15, 104, "MathCaption"],

Cell[CellGroupData[{
Cell[447423, 17761, 583, 11, 122, "Input"],
Cell[448009, 17774, 853, 27, 73, "Output"]
}, Open  ]],
Cell[448877, 17804, 182, 4, 56, "MathCaption"],
Cell[449062, 17810, 132, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[449219, 17817, 130, 3, 26, "Input"],
Cell[449352, 17822, 13361, 460, 89, 5000, 352, "GraphicsData", "PostScript", \
"Graphics"],
Cell[462716, 18284, 155, 4, 25, "Output"]
}, Open  ]],
Cell[462886, 18291, 117, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[463028, 18298, 132, 3, 26, "Input"],
Cell[463163, 18303, 19476, 549, 163, 5376, 370, "GraphicsData", "PostScript", \
"Graphics"],
Cell[482642, 18854, 155, 4, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[482846, 18864, 69, 0, 36, "Subsection"],
Cell[482918, 18866, 130, 3, 22, "Text"],
Cell[483051, 18871, 219, 8, 24, "Text"],
Cell[483273, 18881, 248, 5, 43, "DisplayMath"],
Cell[483524, 18888, 381, 11, 24, "Text"],
Cell[483908, 18901, 368, 8, 63, "DisplayMath"],
Cell[484279, 18911, 250, 6, 56, "MathCaption"],
Cell[484532, 18919, 416, 8, 90, "Input"],
Cell[484951, 18929, 368, 9, 72, "MathCaption"],
Cell[485322, 18940, 63, 1, 26, "Input"],
Cell[485388, 18943, 218, 7, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[485631, 18954, 432, 9, 90, "Input"],
Cell[486066, 18965, 180, 5, 25, "Output"]
}, Open  ]],
Cell[486261, 18973, 54, 0, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[486340, 18977, 92, 2, 26, "Input"],
Cell[486435, 18981, 13327, 428, 184, 4636, 316, "GraphicsData", "PostScript", \
"Graphics"],
Cell[499765, 19411, 155, 4, 25, "Output"]
}, Open  ]],
Cell[499935, 19418, 135, 3, 22, "Text"],
Cell[500073, 19423, 140, 3, 22, "Text"],
Cell[500216, 19428, 169, 4, 56, "MathCaption"],
Cell[500388, 19434, 397, 8, 90, "Input"],
Cell[500788, 19444, 308, 7, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[501121, 19455, 72, 2, 26, "Input"],
Cell[501196, 19459, 16909, 599, 184, 6440, 465, "GraphicsData", "PostScript", \
"Graphics"],
Cell[518108, 20060, 155, 4, 25, "Output"]
}, Open  ]],
Cell[518278, 20067, 162, 4, 36, "Text"],
Cell[518443, 20073, 142, 4, 23, "Text"],
Cell[518588, 20079, 351, 6, 72, "MathCaption"],

Cell[CellGroupData[{
Cell[518964, 20089, 887, 18, 170, "Input"],
Cell[519854, 20109, 196, 5, 25, "Output"]
}, Open  ]],
Cell[520065, 20117, 182, 4, 37, "Text"],
Cell[520250, 20123, 157, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[520432, 20131, 596, 12, 122, "Input"],
Cell[521031, 20145, 196, 5, 25, "Output"]
}, Open  ]],
Cell[521242, 20153, 133, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[521400, 20160, 131, 3, 26, "Input"],
Cell[521534, 20165, 13309, 427, 184, 4593, 314, "GraphicsData", "PostScript", \
"Graphics"],
Cell[534846, 20594, 155, 4, 25, "Output"]
}, Open  ]],
Cell[535016, 20601, 208, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[535249, 20609, 86, 2, 26, "Input"],
Cell[535338, 20613, 17068, 614, 184, 6141, 474, "GraphicsData", "PostScript", \
"Graphics"],
Cell[552409, 21229, 155, 4, 25, "Output"]
}, Open  ]],
Cell[552579, 21236, 283, 8, 39, "Text"],
Cell[552865, 21246, 136, 3, 22, "Text"],
Cell[553004, 21251, 132, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[553161, 21258, 114, 2, 26, "Input"],
Cell[553278, 21262, 319, 5, 113, "Output"]
}, Open  ]],
Cell[553612, 21270, 152, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[553789, 21277, 95, 2, 26, "Input"],
Cell[553887, 21281, 90, 2, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[554026, 21289, 49, 0, 36, "Subsection"],
Cell[554078, 21291, 389, 6, 50, "Text"],
Cell[554470, 21299, 891, 15, 108, "Text"],
Cell[555364, 21316, 527, 8, 64, "Text"],
Cell[555894, 21326, 222, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[556141, 21334, 876, 17, 172, "Input"],
Cell[557020, 21353, 412, 11, 41, "Output"]
}, Open  ]],
Cell[557447, 21367, 285, 6, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[557757, 21377, 317, 6, 58, "Input"],
Cell[558077, 21385, 146924, 2191, 294, 24691, 677, "GraphicsData", \
"PostScript", "Graphics"],
Cell[705004, 23578, 161, 4, 25, "Output"]
}, Open  ]],
Cell[705180, 23585, 512, 12, 72, "MathCaption"],

Cell[CellGroupData[{
Cell[705717, 23601, 996, 20, 188, "Input",
  CellTags->"ELPDEExample"],
Cell[706716, 23623, 442, 12, 41, "Output",
  CellTags->"ELPDEExample"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[707207, 23641, 44, 0, 36, "Subsection"],
Cell[707254, 23643, 125, 3, 22, "Text"],
Cell[707382, 23648, 223, 4, 56, "MathCaption"],
Cell[707608, 23654, 905, 16, 154, "Input"],
Cell[708516, 23672, 157, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[708698, 23680, 495, 8, 106, "Input"],
Cell[709196, 23690, 1618, 41, 115, "Output"]
}, Open  ]],
Cell[710829, 23734, 442, 7, 64, "Text"],
Cell[711274, 23743, 237, 5, 36, "Text"],

Cell[CellGroupData[{
Cell[711536, 23752, 589, 10, 154, "Input"],
Cell[712128, 23764, 674, 20, 51, "Output"]
}, Open  ]],
Cell[712817, 23787, 433, 7, 64, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[713299, 23800, 49, 1, 61, "Section",
  CellTags->"s:4"],
Cell[713351, 23803, 814, 14, 94, "Text"],
Cell[714168, 23819, 302, 6, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[714495, 23829, 356, 7, 67, "Input"],
Cell[714854, 23838, 61, 2, 25, "Output"]
}, Open  ]],
Cell[714930, 23843, 103, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[715058, 23850, 408, 8, 84, "Input"],
Cell[715469, 23860, 62, 2, 25, "Output"]
}, Open  ]],
Cell[715546, 23865, 397, 7, 50, "Text"],
Cell[715946, 23874, 284, 5, 72, "MathCaption"],

Cell[CellGroupData[{
Cell[716255, 23883, 532, 10, 86, "Input"],
Cell[716790, 23895, 64, 2, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[716903, 23903, 52, 1, 61, "Section",
  CellTags->"s:5"],

Cell[CellGroupData[{
Cell[716980, 23908, 83, 1, 36, "Subsection",
  CellTags->"EventLocatorOptionSummary"],
Cell[717066, 23911, 1796, 38, 348, "DefinitionBox3Col"],
Cell[718865, 23951, 46, 0, 15, "Caption"]
}, Open  ]],

Cell[CellGroupData[{
Cell[718948, 23956, 97, 1, 36, "Subsection",
  CellTags->"EventLocationMethodOptionSummary"],
Cell[719048, 23959, 2063, 45, 148, "DefinitionBox"],
Cell[721114, 24006, 63, 0, 15, "Caption"]
}, Open  ]],

Cell[CellGroupData[{
Cell[721214, 24011, 69, 1, 36, "Subsection",
  CellTags->"BrentOptionSummary"],
Cell[721286, 24014, 2810, 61, 284, "DefinitionBox3Col"],
Cell[724099, 24077, 59, 0, 15, "Caption"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

