(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.1'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    924019,      51656]*)
(*NotebookOutlinePosition[    928613,      51796]*)
(*  CellTagsIndexPosition[    927637,      51764]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["ExplicitRungeKutta", "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"s:1"],

Cell["\<\
This loads packages containing some test problems and utility \
functions.\
\>", "Text"],

Cell[BoxData[{
    \(Needs["\<DifferentialEquations`NDSolveProblems`\>"]; \), "\
\[IndentingNewLine]", 
    \(Needs["\<DifferentialEquations`NDSolveUtilities`\>"]; \), "\
\[IndentingNewLine]", 
    \(Needs["\<Graphics`Graphics`\>"]; \)}], "Input",
  CellLabel->"In[4]:=",
  InitializationCell->True],

Cell[CellGroupData[{

Cell["Euler's method", "Subsection"],

Cell["\<\
One of the first and simplest methods for solving initial value \
problems was proposed by Euler:\
\>", "Text"],

Cell[BoxData[
    \(TraditionalForm\`y\_\(n + 1\)\  = \ 
      y\_\(\(n\)\(\ \)\) + \ 
        h\ \(\(f(t\_n, \ y\_n)\)\(.\)\)\)], "NumberedEquation",
  CellTags->"EulersMethod"],

Cell["Euler's method is not very accurate.", "Text"],

Cell[TextData[{
  "Local accuracy is measured by how high terms are matched with the Taylor \
expansion of the solution. Euler's method is first-order accurate, so that \
errors occur one order higher starting at powers of ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(\ \)\(h\^2\)\)\)]],
  "."
}], "Text"],

Cell[TextData[{
  "Euler's method is implemented in ",
  StyleBox["NDSolve", "MR"],
  " as ",
  StyleBox["ExplicitEuler", "MR"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[{\(y'\)[t] \[Equal] \(-y[t]\), \ y[0] \[Equal] 1}, 
      y[t], {t, 0, 
        1}, \[IndentingNewLine]Method \[Rule] "\<ExplicitEuler\>", \ 
      StartingStepSize \[Rule] 1/10]\)], "Input",
  CellLabel->"In[7]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{\(y[t]\), "\[Rule]", 
          RowBox[{
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False], "[", "t", "]"}]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[7]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Generalizing Euler's method", "Subsection"],

Cell["\<\
The idea of Runge-Kutta methods is to take successive (weighted) \
Euler steps to approximate a Taylor series. In this way function evaluations \
(and not derivatives) are used.\
\>", "Text"],

Cell[TextData[{
  "For example, consider the one-step formulation of the ",
  StyleBox["midpoint method",
    FontSlant->"Italic"],
  ":"
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {\(\(k\_1\)\(\ \)\), "=", \(\(\ \)\(f(t\_n, \ y\_n)\)\)},
          {\(k\_2\), 
            "=", \(\(\ \)\(f(
              t\_n + \ \(1\/2\) h, \ \ y\_n\  + \ 1\/2\ \ h\ k\_1\ )\)\)},
          {\(y\_\(n + 1\)\), 
            "=", \(\(\ \)\(y\_n\  + \ h\ k\_\(\(2\)\(\ \)\)\)\)}
          },
        ColumnAlignments->{Center, Center, Left}], 
      TraditionalForm]], "NumberedEquation",
  CellTags->"ExplicitMidpointMethod"],

Cell[TextData[{
  "The midpoint method can be shown to have a local error of ",
  Cell[BoxData[
      \(TraditionalForm\`O(h\^3)\)]],
  " so  it is second-order accurate."
}], "Text"],

Cell[TextData[{
  "The midpoint method is implemented in ",
  StyleBox["NDSolve", "MR"],
  " as ",
  StyleBox["ExplicitMidpoint", "MR"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[{\(y'\)[t] \[Equal] \(-y[t]\), \ y[0] \[Equal] 1}, 
      y[t], {t, 0, 
        1}, \[IndentingNewLine]Method \[Rule] "\<ExplicitMidpoint\>", \ 
      StartingStepSize \[Rule] 1/10]\)], "Input",
  CellLabel->"In[8]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{\(y[t]\), "\[Rule]", 
          RowBox[{
            TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
              False,
              Editable->False], "[", "t", "]"}]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[8]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Runge-Kutta methods", "Subsection"],

Cell[TextData[{
  "Extending the approach in (",
  
  CounterBox["NumberedEquation", "ExplicitMidpointMethod"],
  "), repeated function evaluation can be used to obtain \
higher-and-higher-order methods."
}], "Text"],

Cell[TextData[{
  "Denote the Runge-Kutta method for the approximate solution to an initial \
value problem at ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(\ \ \)\(\(t\_\(n + 1\)\  = \ 
        t\_n\  + \ h\)\(,\)\)\)\)]],
  "  by:"
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {\(g\_i\), \(\(=\)\(\ \)\), \(\(y\_n\  + \ 
              h\ \(\[Sum]\+\(j = 1\)\%s\ a\_\(i, j\)\ k\_j\)\)\(\ \)\(,\)\), 
            " "},
          {\(k\_i\), 
            "=", \(\(f\ \((t\_n\  + \ 
                  c\_i\ h, \ \ g\_i)\)\)\(,\)\), \(i\  = \ 1, \ 
            2, \[Ellipsis], \ s, \)},
          {\(y\_\(n + 1\)\), 
            "=", \(y\_n\  + \ 
              h\ \(\[Sum]\+\(i = 1\)\%s\ \(\(b\_i\)\(\ \)\(k\_i\)\(\ \)\)\)\),
             " "}
          },
        ColumnAlignments->{Center, Center, Left, Center}], 
      TraditionalForm]], "NumberedEquation",
  CellTags->"ERKMethod"],

Cell[TextData[{
  "where ",
  StyleBox["s",
    FontSlant->"Italic"],
  " is the number of stages."
}], "Text"],

Cell[TextData[{
  "It is generally assumed that the ",
  StyleBox["row-sum conditions",
    FontSlant->"Italic"],
  " hold:"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`c\_i\  = \ \[Sum]\+\(i = 1\)\%s\ 
        a\_\(i, j\)\)], "NumberedEquation",
  CellTags->"RowSumConditions"],

Cell["\<\
These effectively determine the points in time at which the \
function is sampled and are a particularly useful device in the derivation of \
high-order Runge-Kutta methods.\
\>", "Text"],

Cell[TextData[{
  "The ",
  StyleBox["coefficients of the method ",
    FontSlant->"Italic"],
  "are ",
  StyleBox["free parameters",
    FontSlant->"Italic"],
  " that are chosen to satisfy a Taylor series expansion through some order \
in the time step ",
  StyleBox["h",
    FontSlant->"Italic"],
  ". In practice other conditions such as stability can also constrain the \
coefficients."
}], "Text"],

Cell[TextData[{
  "Explicit Runge-Kutta methods are a special case where the matrix ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(\ \)\(A\)\(\ \)\)\)]],
  " is strictly lower triangular:"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`a\_\(i, \ j\)\  = \ 0, \ j\  \[GreaterEqual] i, \ 
    j = \ 1, \[Ellipsis], \(\(s\)\(.\)\)\)], "DisplayFormula"],

Cell[TextData[{
  "It has become customary to denote the method coefficients ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          StyleBox["c",
            FontWeight->"Bold"], " ", "=", " ", 
          FormBox[\(\(\([\)\(c\_i\)\(]\)\)\^T\),
            "TraditionalForm"]}], TraditionalForm]]],
  ", ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          StyleBox["b",
            FontWeight->"Bold"], "=", " ", 
          FormBox[\(\(\([\)\(b\_i\)\(]\)\)\^T\),
            "TraditionalForm"]}], TraditionalForm]]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`A\  = \ \(\([\)\(\ \)\(a\_\(i, 
            j\)\)\(\ \)\(]\)\)\)]],
  " using a ",
  StyleBox["Butcher table",
    FontSlant->"Italic"],
  ", which has the following form for explicit Runge-Kutta methods:"
}], "Text"],

Cell[BoxData[
    FormBox[
      TagBox[GridBox[{
            {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
            {\(c\_2\), \(a\_\(2, 1\)\), "0", "\[CenterEllipsis]", "0", "0"},
            {"\[VerticalEllipsis]", "\[VerticalEllipsis]", 
              "\[VerticalEllipsis]", "\[DescendingEllipsis]", 
              "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
            {\(c\_s\), \(a\_\(s, 1\)\), \(a\_\(s, 2\)\), 
              "\[CenterEllipsis]", \(a\_\(s, s - 1\)\), "0"},
            {"", \(b\_1\), \(b\_2\), 
              "\[CenterEllipsis]", \(b\_\(s - 1\)\), \(b\_s\)}
            },
          RowLines->{False, False, False, True},
          ColumnLines->{True, False}],
        DisplayForm], TraditionalForm]], "NumberedEquation",
  CellTags->"ButcherTable"],

Cell["\<\
The row-sum conditions can be visualized as summing across the rows \
of the table.\
\>", "Text"],

Cell[TextData[{
  "Notice that a consequence of explicitness is ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(\ \)\(c\_1\  = \ 0\)\)\)]],
  ", so that the function is sampled at the beginning of the current \
integration step."
}], "Text"],

Cell[CellGroupData[{

Cell["Example", "Subsubsection"],

Cell[TextData[{
  "The Butcher table for the explicit midpoint method (",
  
  CounterBox["NumberedEquation", "ExplicitMidpointMethod"],
  ") is given by:"
}], "Text"],

Cell[BoxData[
    FormBox[
      TagBox[GridBox[{
            {"0", "0", "0"},
            {\(1\/2\), \(1\/2\), "0"},
            {"", "0", "1"}
            },
          RowLines->{False, True},
          ColumnLines->{True, False}],
        DisplayForm], TraditionalForm]], "NumberedEquation",
  CellTags->"ExplicitMidpointButcherTable"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["FSAL schemes", "Subsection"],

Cell["\<\
A particularly interesting special class of explicit Runge-Kutta \
methods, used in most modern codes, are those for which the coefficient have \
a special structure known as First Same As Last (FSAL):\
\>", "Text"],

Cell[BoxData[
    \(TraditionalForm\`a\_\(s, i\)\  = \ b\_i, \ i\  = \ 1, \[Ellipsis], 
    s - 1\ \ and\ \ b\_s = \ 0. \)], "NumberedEquation",
  CellTags->"FSALConditions"],

Cell[TextData[{
  "For consistent FSAL schemes the Butcher table ",
  "(",
  
  CounterBox["NumberedEquation", "ButcherTable"],
  ") ",
  "has the form:"
}], "Text"],

Cell[BoxData[
    FormBox[
      TagBox[GridBox[{
            {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
            {\(c\_2\), \(a\_\(2, 1\)\), "0", "\[CenterEllipsis]", "0", "0"},
            {"\[VerticalEllipsis]", "\[VerticalEllipsis]", 
              "\[VerticalEllipsis]", "\[DescendingEllipsis]", 
              "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
            {\(c\_\(s - 1\)\), \(a\_\(s - 1, 1\)\), \(a\_\(s - 1, 2\)\), 
              "\[CenterEllipsis]", "0", "0"},
            {"1", \(b\_1\), \(b\_2\), "\[CenterEllipsis]", \(b\_\(s - 1\)\), 
              "0"},
            {"", \(b\_1\), \(b\_2\), "\[CenterEllipsis]", \(b\_\(s - 1\)\), 
              "0"}
            },
          RowLines->{False, False, False, False, True},
          ColumnLines->{True, False}],
        DisplayForm], TraditionalForm]], "NumberedEquation",
  CellTags->"FSALButcherTable"],

Cell[TextData[{
  "The advantage of FSAL methods is that the function value ",
  Cell[BoxData[
      \(TraditionalForm\`k\_s\)]],
  " at the end of one integration step is the same as the first function \
value ",
  Cell[BoxData[
      \(TraditionalForm\`k\_1\)]],
  " at the next integration step."
}], "Text"],

Cell[TextData[{
  "The function values at the beginning and end of each integration step are \
required anyway when constructing the ",
  StyleBox["InterpolatingFunction", "MR"],
  " that is used for dense output in ",
  StyleBox["NDSolve", "MR"],
  "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Embedded pairs and local error estimation", "Subsection"],

Cell[TextData[{
  "An efficient means of obtaining local error estimates for adaptive step \
size control is to consider two methods of different orders ",
  Cell[BoxData[
      \(TraditionalForm\`p\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\(p\&^\)\)]],
  " that share the same coefficient matrix (and hence function values)."
}], "Text"],

Cell[BoxData[
    FormBox[
      TagBox[GridBox[{
            {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
            {\(c\_2\), \(a\_\(2, 1\)\), "0", "\[CenterEllipsis]", "0", "0"},
            {"\[VerticalEllipsis]", "\[VerticalEllipsis]", 
              "\[VerticalEllipsis]", "\[DescendingEllipsis]", "0", 
              "\[VerticalEllipsis]"},
            {\(c\_\(s - 1\)\), \(a\_\(s - 1, 1\)\), \(a\_\(s - 1, 2\)\), 
              "\[CenterEllipsis]", "0", "0"},
            {\(c\_s\), \(a\_\(s, 1\)\), \(a\_\(s, 2\)\), 
              "\[CenterEllipsis]", \(a\_\(s, s - 1\)\), "0"},
            {"", \(b\_1\), \(b\_2\), 
              "\[CenterEllipsis]", \(b\_\(s - 1\)\), \(b\_s\)},
            {" ", \(\(b\&^\)\_1\), \(\(b\&^\)\_2\), 
              "\[CenterEllipsis]", \(\(b\&^\)\_\(s - 1\)\), \(\(b\&^\)\_s\)}
            },
          RowLines->{False, False, False, False, True, False},
          ColumnLines->{True, False}],
        DisplayForm], TraditionalForm]], "NumberedEquation",
  CellTags->"EmbeddedButcherTable"],

Cell["These give two solutions:", "Text"],

Cell[BoxData[
    \(TraditionalForm\`y\_\(n + 1\)\  = \ 
      y\_n\  + \ 
        h\ \(\[Sum]\+\(i = 1\)\%s\ b\_i\ k\_i\)\)], "NumberedEquation",
  CellTags->"HigherOrder"],

Cell[BoxData[
    \(TraditionalForm\`\(y\&^\)\_\(n + 1\)\  = \ 
      y\_n\  + \ 
        h\ \(\[Sum]\+\(i = 1\)\%s\ \(b\&^\)\_i\ k\_i\)\)], "NumberedEquation",\

  CellTags->"LowerOrder"],

Cell[TextData[{
  "A commonly used notation is ",
  Cell[BoxData[
      FormBox[
        RowBox[{"p", "(", 
          FormBox[\(p\&^\),
            "TraditionalForm"], ")"}], TraditionalForm]]],
  ", typically with ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          FormBox[\(p\&^\),
            "TraditionalForm"], "=", \(p - 1\)}], TraditionalForm]]],
  " or ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          FormBox[\(p\&^\),
            "TraditionalForm"], "=", \(p + 1\)}], TraditionalForm]]],
  "."
}], "Text"],

Cell[TextData[{
  "In most modern codes, including the default choice in ",
  StyleBox["NDSolve", "MR"],
  ", the solution is advanced with the more accurate formula so that ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          FormBox[\(p\&^\),
            "TraditionalForm"], "=", \(p - 1\)}], TraditionalForm]]],
  ", which is known as local extrapolation."
}], "Text"],

Cell[TextData[{
  "The vector of coefficients ",
  Cell[BoxData[
      FormBox[
        SuperscriptBox[
          FormBox[
            RowBox[{"e", " ", "=", " ", 
              RowBox[{"[", 
                RowBox[{
                  RowBox[{
                    FormBox[\(b\_1\),
                      "TraditionalForm"], "-", 
                    FormBox[\(\(b\&^\)\_1\),
                      "TraditionalForm"]}], ",", " ", 
                  RowBox[{
                    FormBox[\(b\_2\),
                      "TraditionalForm"], "-", 
                    FormBox[\(\(b\&^\)\_2\),
                      "TraditionalForm"]}], ",", "\[Ellipsis]", ",", " ", 
                  RowBox[{
                    FormBox[\(b\_s\),
                      "TraditionalForm"], "-", 
                    FormBox[\(\(b\&^\)\_s\),
                      "TraditionalForm"]}]}], "]"}]}],
            "TraditionalForm"], "T"], TraditionalForm]]],
  " gives an error estimator avoiding subtractive cancellation of ",
  Cell[BoxData[
      \(TraditionalForm\`y\_n\)]],
  " in floating-point arithmetic when forming the difference between (",
  
  CounterBox["NumberedEquation", "HigherOrder"],
  ") and (",
  
  CounterBox["NumberedEquation", "LowerOrder"],
  ")."
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`err\_n\  = \ \ h\ \(\[Sum]\+\(i = 1\)\%s\ 
          e\_i\ k\_i\)\)], "DisplayFormula"],

Cell[TextData[{
  "The quantity ",
  Cell[BoxData[
      \(TraditionalForm\`\[LeftDoubleBracketingBar]\ 
        err\_n\[RightDoubleBracketingBar]\)]],
  " gives a scalar measure of the error that can be used for step size \
selection."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Step control", "Subsection"],

Cell["\<\
The classical, Integral or I step size controller uses the formula:\
\
\>", "Text"],

Cell[BoxData[
    \(TraditionalForm\`h\_\(n + 1\) = \ 
      h\_n\ \((Tol\/\[LeftDoubleBracketingBar]err\_n\
\[RightDoubleBracketingBar])\)\^\(1/\(p\& \[Tilde] \)\)\)], "NumberedEquation",\

  CellTags->"IController"],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          FormBox[\(p\& \[Tilde] \),
            "TraditionalForm"], " ", "=", " ", 
          RowBox[{
            FormBox[\(min(\(p\&^\), \ p)\),
              "TraditionalForm"], " ", "+", " ", "1"}]}], TraditionalForm]]],
  "."
}], "Text"],

Cell["\<\
The error estimate is therefore used to determine the next step \
size to use from the current step size.\
\>", "Text"],

Cell[TextData[{
  "The notation ",
  Cell[BoxData[
      \(TraditionalForm\`Tol/\[LeftDoubleBracketingBar]err\_n\
\[RightDoubleBracketingBar]\)]],
  " is explained within ",
  StyleBox[ButtonBox["Norm",
    ButtonData:>{"VectorNorm.nb", None},
    ButtonStyle->"Hyperlink"], "MR"],
  " and ",
  StyleBox[ButtonBox["ScaledVectorNorm",
    ButtonData:>{"VectorNorm.nb", None},
    ButtonStyle->"Hyperlink"], "MR"],
  "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Overview", "Subsection"],

Cell["\<\
Explicit Runge Kutta pairs of orders 2(1) through 9(8) have been \
implemented.\
\>", "Text"],

Cell["Formula pairs have the following properties:", "Text"],

Cell["First Same As Last strategy.", "BulletedList"],

Cell["\<\
Local extrapolation mode, that is the higher-order formula is used \
to propagate the solution.\
\>", "BulletedList"],

Cell[TextData[{
  "Stiffness detection capability (see [",
  ButtonBox["HW96",
    ButtonData:>{"References.nb", "HW96"},
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["P83",
    ButtonData:>{"References.nb", "P83"},
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["R87",
    ButtonData:>{"References.nb", "R87"},
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["S84",
    ButtonData:>{"References.nb", "S84"},
    ButtonStyle->"Hyperlink"],
  "])."
}], "BulletedList"],

Cell[TextData[{
  "Proportional-Integral step size controller for stiff and quasi-stiff \
systems [",
  ButtonBox["G91",
    ButtonData:>{"References.nb", "G91"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "BulletedList"],

Cell[TextData[{
  "Optimal formula pairs of orders 2(1), 3(2), and 4(3) subject to the above \
requirements have been derived using ",
  StyleBox["Mathematica,",
    FontSlant->"Italic"],
  " and are described in [",
  ButtonBox["SS04b",
    ButtonData:>{"References.nb", "SS04b"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell[TextData[{
  "The 5(4) pair selected is due to Bogacki and Shampine [",
  ButtonBox["BS89b",
    ButtonData:>{"References.nb", "BS89b"},
    ButtonStyle->"Hyperlink"],
  ", ",
  ButtonBox["S94",
    ButtonData:>{"References.nb", "S94"},
    ButtonStyle->"Hyperlink"],
  "] and the 6(5), 7(6), 8(7), and 9(8) pairs are due to Verner."
}], "Text"],

Cell[TextData[{
  "For the selection of higher-order pairs, issues such as local truncation \
error ratio and stability region compatibility should be considered (see [",
  ButtonBox["S94",
    ButtonData:>{"References.nb", "S94"},
    ButtonStyle->"Hyperlink"],
  "]) . Various tools have been written to assess these qualitative \
features."
}], "Text"],

Cell["\<\
Methods are interchangeable so that, for example, it is possible to \
substitute the 5(4) method of Bogacki and Shampine with a method of Dormand \
and Prince.\
\>", "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example", "Section",
  CellTags->"s:2"],

Cell["\<\
Define the Brusselator ODE problem, which models a chemical \
reaction.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(system\  = \ GetNDSolveProblem["\<BrusselatorODE\>"]\)], "Input",
  CellLabel->"In[9]:="],

Cell[BoxData[
    RowBox[{"NDSolveProblem", "[", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{"{", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_1\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(1 - 4\ Y\_1[T] + Y\_1[T]\^2\ Y\_2[T]\)}], ",", 
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_2\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(3\ Y\_1[T] - Y\_1[T]\^2\ Y\_2[T]\)}]}], "}"}], 
          ",", \({Y\_1[0] \[Equal] 3\/2, Y\_2[0] \[Equal] 3}\), 
          ",", \({Y\_1[T], Y\_2[T]}\), ",", \({T, 0, 20}\), ",", \({}\), 
          ",", \({}\)}], "}"}], "]"}]], "Output",
  CellLabel->"Out[9]="]
}, Open  ]],

Cell["This solves the system using an explicit Runge-Kutta method.", \
"MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol\  = \ 
      NDSolve[system, Method \[Rule] "\<ExplicitRungeKutta\>"]\)], "Input",
  CellLabel->"In[10]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[10]="]
}, Open  ]],

Cell["Extract the interpolating functions from the solution.", "MathCaption"],

Cell[BoxData[
    \(ifuns\  = \ 
      system["\<DependentVariables\>"[]]\  /. First[sol]; \)], "Input",
  CellLabel->"In[11]:="],

Cell["Plot the solution components.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ParametricPlot[Evaluate[ifuns], 
      Evaluate[system["\<TimeData\>"[]]]]; \)], "Input",
  CellLabel->"In[12]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1.0327 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.254211 -0.192322 0.254211 [
[.15091 .04939 -9 -9 ]
[.15091 .04939 9 0 ]
[.27802 .04939 -3 -9 ]
[.27802 .04939 3 0 ]
[.40513 .04939 -9 -9 ]
[.40513 .04939 9 0 ]
[.53223 .04939 -3 -9 ]
[.53223 .04939 3 0 ]
[.65934 .04939 -9 -9 ]
[.65934 .04939 9 0 ]
[.78644 .04939 -3 -9 ]
[.78644 .04939 3 0 ]
[.91355 .04939 -9 -9 ]
[.91355 .04939 9 0 ]
[.01131 .3161 -6 -4.5 ]
[.01131 .3161 0 4.5 ]
[.01131 .57031 -6 -4.5 ]
[.01131 .57031 0 4.5 ]
[.01131 .82452 -6 -4.5 ]
[.01131 .82452 0 4.5 ]
[ 0 0 0 0 ]
[ 1 1.0327 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.15091 .06189 m
.15091 .06814 L
s
[(0.5)] .15091 .04939 0 1 Mshowa
.27802 .06189 m
.27802 .06814 L
s
[(1)] .27802 .04939 0 1 Mshowa
.40513 .06189 m
.40513 .06814 L
s
[(1.5)] .40513 .04939 0 1 Mshowa
.53223 .06189 m
.53223 .06814 L
s
[(2)] .53223 .04939 0 1 Mshowa
.65934 .06189 m
.65934 .06814 L
s
[(2.5)] .65934 .04939 0 1 Mshowa
.78644 .06189 m
.78644 .06814 L
s
[(3)] .78644 .04939 0 1 Mshowa
.91355 .06189 m
.91355 .06814 L
s
[(3.5)] .91355 .04939 0 1 Mshowa
.125 Mabswid
.04923 .06189 m
.04923 .06564 L
s
.07465 .06189 m
.07465 .06564 L
s
.10007 .06189 m
.10007 .06564 L
s
.12549 .06189 m
.12549 .06564 L
s
.17634 .06189 m
.17634 .06564 L
s
.20176 .06189 m
.20176 .06564 L
s
.22718 .06189 m
.22718 .06564 L
s
.2526 .06189 m
.2526 .06564 L
s
.30344 .06189 m
.30344 .06564 L
s
.32886 .06189 m
.32886 .06564 L
s
.35428 .06189 m
.35428 .06564 L
s
.3797 .06189 m
.3797 .06564 L
s
.43055 .06189 m
.43055 .06564 L
s
.45597 .06189 m
.45597 .06564 L
s
.48139 .06189 m
.48139 .06564 L
s
.50681 .06189 m
.50681 .06564 L
s
.55765 .06189 m
.55765 .06564 L
s
.58307 .06189 m
.58307 .06564 L
s
.60849 .06189 m
.60849 .06564 L
s
.63392 .06189 m
.63392 .06564 L
s
.68476 .06189 m
.68476 .06564 L
s
.71018 .06189 m
.71018 .06564 L
s
.7356 .06189 m
.7356 .06564 L
s
.76102 .06189 m
.76102 .06564 L
s
.81186 .06189 m
.81186 .06564 L
s
.83728 .06189 m
.83728 .06564 L
s
.86271 .06189 m
.86271 .06564 L
s
.88813 .06189 m
.88813 .06564 L
s
.93897 .06189 m
.93897 .06564 L
s
.96439 .06189 m
.96439 .06564 L
s
.98981 .06189 m
.98981 .06564 L
s
.25 Mabswid
0 .06189 m
1 .06189 L
s
.02381 .3161 m
.03006 .3161 L
s
[(2)] .01131 .3161 1 0 Mshowa
.02381 .57031 m
.03006 .57031 L
s
[(3)] .01131 .57031 1 0 Mshowa
.02381 .82452 m
.03006 .82452 L
s
[(4)] .01131 .82452 1 0 Mshowa
.125 Mabswid
.02381 .11273 m
.02756 .11273 L
s
.02381 .16357 m
.02756 .16357 L
s
.02381 .21442 m
.02756 .21442 L
s
.02381 .26526 m
.02756 .26526 L
s
.02381 .36694 m
.02756 .36694 L
s
.02381 .41778 m
.02756 .41778 L
s
.02381 .46863 m
.02756 .46863 L
s
.02381 .51947 m
.02756 .51947 L
s
.02381 .62115 m
.02756 .62115 L
s
.02381 .672 m
.02756 .672 L
s
.02381 .72284 m
.02756 .72284 L
s
.02381 .77368 m
.02756 .77368 L
s
.02381 .01105 m
.02756 .01105 L
s
.02381 .87536 m
.02756 .87536 L
s
.02381 .92621 m
.02756 .92621 L
s
.02381 .97705 m
.02756 .97705 L
s
.02381 1.02789 m
.02756 1.02789 L
s
.25 Mabswid
.02381 0 m
.02381 1.0327 L
s
0 0 m
1 0 L
1 1.0327 L
0 1.0327 L
closepath
clip
newpath
.5 Mabswid
.40513 .57031 m
.41634 .55584 L
.42813 .54051 L
.45329 .50738 L
.50869 .4321 L
.56535 .35 L
.61241 .27231 L
.62996 .23647 L
.63623 .22075 L
.64072 .20666 L
.64348 .19422 L
.64458 .18338 L
.64413 .17408 L
.64223 .16626 L
.63901 .15984 L
.63461 .15469 L
.62285 .14778 L
.61575 .1458 L
.60799 .14468 L
.59967 .14433 L
.59089 .14467 L
.58237 .14552 L
.57359 .14686 L
.55542 .15079 L
.51786 .16265 L
.44332 .197 L
.31638 .28529 L
.21786 .38992 L
.16276 .48068 L
.13331 .56657 L
.12204 .64135 L
.12005 .67409 L
.11955 .69007 L
.11941 .69797 L
.11933 .70581 L
.11931 .71004 L
.1193 .71426 L
.11932 .71846 L
.11934 .72264 L
.11944 .73096 L
.11959 .73922 L
.12067 .77166 L
.12438 .82963 L
.12997 .88271 L
.13867 .93469 L
.15067 .9753 L
.15984 .99233 L
.1655 .99897 L
.17219 1.00392 L
.17594 1.00562 L
Mistroke
.17794 1.00626 L
.18003 1.00676 L
.18223 1.0071 L
.18454 1.00726 L
.18697 1.00724 L
.18955 1.00702 L
.19227 1.00657 L
.19515 1.00589 L
.20145 1.00376 L
.21669 .99578 L
.35278 .86336 L
.62136 .55548 L
.84601 .28258 L
.92985 .16629 L
.97058 .0898 L
.97619 .067 L
.97529 .05066 L
.96938 .03942 L
.95979 .03205 L
.94756 .02759 L
.9335 .02529 L
.91819 .02459 L
.90209 .02508 L
.88549 .02647 L
.86863 .02854 L
.83471 .03413 L
.76831 .04884 L
.63787 .08865 L
.52684 .13486 L
.3565 .23843 L
.24591 .34136 L
.17263 .44545 L
.13813 .52949 L
.12344 .60192 L
.11994 .63845 L
.11895 .65619 L
.11834 .67362 L
.11806 .68965 L
.11803 .69362 L
.11801 .69757 L
.11801 .70152 L
.11802 .70545 L
.11804 .70936 L
.11808 .71327 L
.11818 .72104 L
.11852 .73642 L
.11978 .7691 L
.12163 .80084 L
.12685 .86016 L
.13378 .91106 L
Mistroke
.14455 .95932 L
.1601 .99377 L
.17135 1.00434 L
.17843 1.00728 L
.1804 1.0077 L
.18246 1.00798 L
.18462 1.00811 L
.1869 1.00808 L
.1895 1.00784 L
.19225 1.00738 L
.19823 1.00575 L
.21265 .99901 L
.25818 .96296 L
.57844 .60641 L
.90661 .02465 L
.64948 .08451 L
.4571 .17147 L
.30668 .27989 L
.21297 .38219 L
.15528 .48146 L
.12943 .56507 L
.12021 .63461 L
.11843 .67012 L
.11808 .68744 L
.11804 .69172 L
.11802 .696 L
.11801 .70025 L
.11801 .70449 L
.11803 .70844 L
.11807 .71237 L
.11817 .72019 L
.1185 .73569 L
.11963 .76609 L
.12332 .82331 L
.12937 .8813 L
.13736 .93033 L
.15009 .97508 L
.15057 .97623 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[12]:=",
  ImageSize->{288, 297.375},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0001:B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`002@3oool00`000000oooo0?ooo`3o0?oooaD0oooo00090?ooo`030000
003oool0oooo0?l0oooo5@3oool000T0oooo00<000000?ooo`3oool0o`3ooolE0?ooo`002@3oool2
00000?l0oooo5P3oool000T0oooo00<000000?ooo`3oool0o`3ooolE0?ooo`002@3oool00`000000
oooo0?ooo`3o0?oooaD0oooo00090?ooo`030000003oool0oooo0?l0oooo5@3oool000T0oooo00<0
00000?ooo`3oool06P3oool2000000@0oooo00<000000?ooo`3oool00P3oool3000001/0oooo1000
000J0?ooo`@000000`3oool00`000000oooo0?ooo`020?ooo`<00000703oool4000001X0oooo1000
00030?ooo`030000003oool0oooo0080oooo0`00000L0?ooo`8000006@3ooolE000000l0oooo0009
0?ooo`030000003oool0oooo01T0oooo00@000000?ooo`3oool000002`3oool00`000000oooo0?oo
o`0J0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?oo
o`0I0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool0303oool00`000000oooo0?oo
o`0H0?ooo`040000003oool0oooo00000140oooo1`0000020?ooo`@0000000<0oooo0000003oool0
203oool010000000oooo0?ooo`3oool2000000d0oooo00090?ooo`030000003oool0oooo01T0oooo
00@000000?ooo`3oool000002`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo01/0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo01/0
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo00P0
oooo1`0000050?ooo`@000000`3oool00`000000oooo000000090?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool02P3oool000T0oooo00<000000?ooo`3oool06@3oool010000000oooo
0?ooo`0000080?ooo`<000007@3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo00L0
oooo0`00000N0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool01`3oool3000001`0
oooo0P0000060?ooo`D000001`3oool5000000D0oooo0P0000000`3oool000000?ooo`060?ooo`<0
00001`3oool2000000X0oooo00090?ooo`030000003oool0oooo01T0oooo00@000000?ooo`3oool0
0000203oool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3o
ool01P3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0000000<00000203oool4000000`0
oooo0P0000070?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool01`3oool000T0oooo
00<000000?ooo`3oool06@3oool010000000oooo0?ooo`0000080?ooo`030000003oool0oooo01/0
oooo0`00000K0?ooo`<000002@3oool00`000000oooo0?ooo`0L0?ooo`040000003oool0oooo0000
01X0oooo00@000000?ooo`3oool00000203oool00`000000oooo0?ooo`0K0?ooo`D00000203oool4
000000d0oooo00@000000?ooo`3oool00000203oool00`000000oooo0?ooo`090?ooo`030000003o
ool0oooo00L0oooo00090?ooo`030000003oool0oooo01X0oooo0P0000090?ooo`@00000703oool0
0`000000oooo0?ooo`0K0?ooo`030000003oool0oooo00L0oooo1000000L0?ooo`800000703oool2
000000T0oooo1000000F0?ooo`@0000000<0oooo0000000000001P3oool400000180oooo0`000008
0?ooo`@000002@3oool00`000000oooo0?ooo`060?ooo`002@3oool00`000000oooo0?ooo`3=0?oo
o`<00000203oool5000001L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`060?oo
o`002@3oool00`000000oooo0?ooo`3:0?ooo`<000001`3oool4000001/0oooo00<000000?ooo`3o
ool05@3oool00`000000oooo0?ooo`060?ooo`002@3oool00`000000oooo0?ooo`370?ooo`<00000
1P3oool4000001l0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`060?ooo`000P3o
oooo000001l0000000090?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`@000000`3oool4000000@0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`8000001P3oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo00@0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0oooo0040oooo0009
0?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`0Q
0?ooo`030000003oool0oooo0280oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`08
0?ooo`@000000P3oool4000000l0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool01@3oool000T0oooo00<000000?oo
o`3oool0^P3oool300000080oooo1000000^0?ooo`030000003oool0oooo01P0oooo00<000000?oo
o`3oool01@3oool000T0oooo00<000000?ooo`3oool0]`3oool800000340oooo00<000000?ooo`3o
ool0603oool00`000000oooo0?ooo`060?ooo`002@3oool00`000000oooo0?ooo`2d0?ooo`H00000
=P3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00H0oooo00090?ooo`030000003o
ool0oooo0;40oooo1@00000i0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool01P3o
ool000T0oooo00<000000?ooo`3oool0[P3oool4000003`0oooo00<000000?ooo`3oool06@3oool0
0`000000oooo0?ooo`070?ooo`002@3oool00`000000oooo0?ooo`2[0?ooo`<00000@03oool00`00
0000oooo0?ooo`0I0?ooo`030000003oool0oooo00L0oooo00090?ooo`030000003oool0oooo0:T0
oooo0`0000110?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool01`3oool000T0oooo
00<000000?ooo`3oool0YP3oool4000004<0oooo00<000000?ooo`3oool06@3oool00`000000oooo
0?ooo`080?ooo`002@3oool00`000000oooo0?ooo`2T0?ooo`<00000A@3oool00`000000oooo0?oo
o`0J0?ooo`030000003oool0oooo00P0oooo00090?ooo`030000003oool0oooo0:40oooo10000017
0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool02@3oool000T0oooo00<000000?oo
o`3oool0W`3oool4000004P0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`090?oo
o`002@3oool00`000000oooo0?ooo`2L0?ooo`D00000B@3oool00`000000oooo0?ooo`0J0?ooo`03
0000003oool0oooo00X0oooo00090?ooo`800000V`3oool4000004`0oooo00<000000?ooo`3oool0
6P3oool00`000000oooo0?ooo`0:0?ooo`002@3oool00`000000oooo0?ooo`2H0?ooo`@00000C@3o
ool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo00/0oooo00090?ooo`030000003oool0
oooo09D0oooo1@00001?0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool02`3oool0
00T0oooo00<000000?ooo`3oool0T`3oool400000540oooo00<000000?ooo`3oool06P3oool00`00
0000oooo0?ooo`0<0?ooo`002@3oool00`000000oooo0?ooo`2@0?ooo`D00000D`3oool00`000000
oooo0?ooo`0J0?ooo`030000003oool0oooo00`0oooo00090?ooo`030000003oool0oooo08h0oooo
0P0000000`3oool000000000001D0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0
3@3oool000T0oooo00<000000?ooo`3oool0S03oool5000005H0oooo00<000000?ooo`3oool06P3o
ool00`000000oooo0?ooo`0=0?ooo`002@3oool00`000000oooo0?ooo`2:0?ooo`@00000F03oool0
0`000000oooo0?ooo`0J0?ooo`030000003oool0oooo00h0oooo00090?ooo`030000003oool0oooo
08P0oooo1000001I0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool03P3oool000T0
oooo00<000000?ooo`3oool0Q`3oool3000000d0oooo3`00000o0?ooo`030000003oool0oooo01X0
oooo00<000000?ooo`3oool03`3oool000T0oooo00<000000?ooo`3oool0Q@3oool2000000`0oooo
1000000?0?ooo`800000?03oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo00l0oooo
00090?ooo`030000003oool0oooo08<0oooo0P00000;0?ooo`<000005@3oool2000003X0oooo00<0
00000?ooo`3oool06P3oool00`000000oooo0?ooo`0@0?ooo`002@3oool00`000000oooo0?ooo`21
0?ooo`8000002`3oool2000001X0oooo0P00000g0?ooo`030000003oool0oooo01/0oooo00<00000
0?ooo`3oool0403oool000T0oooo00<000000?ooo`3oool0OP3oool4000000T0oooo0`00000N0?oo
o`030000003oool0oooo03@0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`0A0?oo
o`002@3oool2000007d0oooo100000080?ooo`<000008@3oool00`000000oooo0?ooo`0c0?ooo`03
0000003oool0oooo01/0oooo00<000000?ooo`3oool04@3oool000T0oooo00<000000?ooo`3oool0
NP3oool2000000030?ooo`000000oooo00L0oooo0P00000U0?ooo`030000003oool0oooo0340oooo
00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0B0?ooo`002@3oool00`000000oooo0?oo
o`1h0?ooo`80000000<0oooo0000000000001`3oool2000002L0oooo00<000000?ooo`3oool0<@3o
ool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo01<0oooo00090?ooo`030000003oool0
oooo07H0oooo0P0000000`3oool00000000000070?ooo`800000:@3oool00`000000oooo0?ooo`0`
0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool04`3oool000T0oooo00<000000?oo
o`3oool0M@3oool010000000oooo0?ooo`0000070?ooo`800000:`3oool00`000000oooo0?ooo`0`
0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0503oool000T0oooo00<000000?oo
o`3oool0M03oool010000000oooo0000000000060?ooo`800000;@3oool00`000000oooo0?ooo`0_
0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool05@3oool000T0oooo00<000000?oo
o`3oool0LP3oool4000000H0oooo0P00000_0?ooo`030000003oool0oooo02l0oooo00<000000?oo
o`3oool06@3oool00`000000oooo0?ooo`0F0?ooo`002@3oool00`000000oooo0?ooo`1a0?ooo`03
0000003oool0000000H0oooo0P00000a0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3o
ool06P3oool00`000000oooo0?ooo`0F0?ooo`002@3oool00`000000oooo0?ooo`1_0?ooo`@00000
1@3oool2000003<0oooo00<000000?ooo`3oool0;P3oool00`000000oooo0?ooo`0I0?ooo`030000
003oool0oooo01L0oooo00090?ooo`030000003oool0oooo06h0oooo00<000000?ooo`0000001@3o
ool2000003D0oooo00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`0I0?ooo`030000003o
ool0oooo01P0oooo00090?ooo`030000003oool0oooo06d0oooo0`0000040?ooo`800000=`3oool0
0`000000oooo0?ooo`0/0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool06@3oool0
00T0oooo00<000000?ooo`3oool0J`3oool3000000@0oooo0P00000h0?ooo`030000003oool0oooo
02d0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0I0?ooo`002@3oool00`000000
oooo0?ooo`1Z0?ooo`8000001@3oool00`000000oooo0?ooo`0h0?ooo`030000003oool0oooo02`0
oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0J0?ooo`002@3oool00`000000oooo
0?ooo`1Y0?ooo`800000103oool2000003/0oooo00<000000?ooo`3oool0;03oool00`000000oooo
0?ooo`0H0?ooo`030000003oool0oooo01/0oooo00090?ooo`800000J03oool2000000D0oooo00<0
00000?ooo`3oool0>`3oool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo01T0oooo00<0
00000?ooo`3oool06`3oool000T0oooo00<000000?ooo`3oool0IP3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo03/0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0H
0?ooo`030000003oool0oooo01`0oooo00090?ooo`030000003oool0oooo06@0oooo0P0000050?oo
o`800000?P3oool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo01P0oooo00<000000?oo
o`3oool07@3oool000T0oooo00<000000?ooo`3oool0HP3oool2000000H0oooo00<000000?ooo`3o
ool0?P3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3o
ool07P3oool000T0oooo00<000000?ooo`3oool0H@3oool2000000D0oooo0P0000100?ooo`030000
003oool0oooo02/0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0N0?ooo`002@3o
ool00`000000oooo0?ooo`1O0?ooo`<000001@3oool00`000000oooo0?ooo`100?ooo`030000003o
ool0oooo02X0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0O0?ooo`002@3oool0
0`000000oooo0?ooo`1M0?ooo`<000001@3oool200000480oooo00<000000?ooo`3oool0:`3oool0
0`000000oooo0?ooo`0G0?ooo`030000003oool0oooo0200oooo00090?ooo`030000003oool0oooo
05`0oooo0`0000050?ooo`030000003oool0oooo0480oooo00<000000?ooo`3oool0:P3oool00`00
0000oooo0?ooo`0G0?ooo`030000003oool0oooo0240oooo00090?ooo`030000003oool0oooo05/0
oooo0P0000050?ooo`800000A@3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01L0
oooo00<000000?ooo`3oool08@3oool000T0oooo00<000000?ooo`3oool0FP3oool2000000D0oooo
00<000000?ooo`3oool0A03oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01L0oooo
00<000000?ooo`3oool08P3oool000T0oooo00<000000?ooo`3oool0F@3oool2000000D0oooo00<0
00000?ooo`3oool0A@3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo01L0oooo00<0
00000?ooo`3oool08`3oool000T0oooo00<000000?ooo`3oool0F03oool00`000000oooo0?ooo`03
0?ooo`800000A`3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01L0oooo00<00000
0?ooo`3oool08`3oool000T0oooo00<000000?ooo`3oool0E`3oool00`000000oooo0?ooo`030?oo
o`030000003oool0oooo04L0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0G0?oo
o`030000003oool0oooo02@0oooo00090?ooo`030000003oool0oooo05D0oooo0P0000040?ooo`80
0000B@3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3o
ool09@3oool000T0oooo00<000000?ooo`3oool0E03oool2000000@0oooo00<000000?ooo`3oool0
B@3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0
9P3oool000T0oooo0P00001C0?ooo`800000103oool2000004/0oooo00<000000?ooo`3oool0:P3o
ool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo02H0oooo00090?ooo`030000003oool0
oooo0540oooo0P0000040?ooo`030000003oool0oooo04/0oooo00<000000?ooo`3oool0:@3oool0
0`000000oooo0?ooo`0F0?ooo`030000003oool0oooo02L0oooo00090?ooo`030000003oool0oooo
0500oooo0P0000040?ooo`030000003oool0oooo04/0oooo00<000000?ooo`3oool0:P3oool00`00
0000oooo0?ooo`0E0?ooo`030000003oool0oooo02P0oooo00090?ooo`030000003oool0oooo04h0
oooo0`0000030?ooo`800000C@3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01D0
oooo00<000000?ooo`3oool0:@3oool000T0oooo00<000000?ooo`3oool0C@3oool00`000000oooo
000000030?ooo`030000003oool0oooo04d0oooo00<000000?ooo`3oool0:@3oool00`000000oooo
0?ooo`0F0?ooo`030000003oool0oooo02T0oooo00090?ooo`030000003oool0oooo04`0oooo00<0
00000?ooo`0000000P3oool2000004l0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?oo
o`0E0?ooo`030000003oool0oooo02X0oooo00090?ooo`030000003oool0oooo04/0oooo00<00000
0?ooo`0000000P3oool00`000000oooo0?ooo`1?0?ooo`030000003oool0oooo02T0oooo00<00000
0?ooo`3oool05@3oool00`000000oooo0?ooo`0[0?ooo`002@3oool00`000000oooo0?ooo`1:0?oo
o`030000003oool000000080oooo00<000000?ooo`3oool0C`3oool00`000000oooo0?ooo`0Z0?oo
o`030000003oool0oooo01@0oooo00<000000?ooo`3oool0;03oool000T0oooo00<000000?ooo`3o
ool0B@3oool00`000000oooo000000020?ooo`030000003oool0oooo04l0oooo00<000000?ooo`3o
ool0:P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02`0oooo00090?ooo`030000
003oool0oooo04T0oooo0P0000020?ooo`030000003oool0oooo0500oooo00<000000?ooo`3oool0
:P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo02d0oooo00090?ooo`030000003o
ool0oooo04P0oooo0P0000020?ooo`030000003oool0oooo0500oooo00<000000?ooo`3oool0:P3o
ool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo02h0oooo00090?ooo`030000003oool0
oooo04L0oooo0P0000020?ooo`030000003oool0oooo0540oooo00<000000?ooo`3oool0:@3oool0
0`000000oooo0?ooo`0D0?ooo`030000003oool0oooo02l0oooo00000`3oool00000000000020000
00@0oooo00<000000?ooo`3oool0AP3oool200000080oooo00<000000?ooo`3oool0D@3oool00`00
0000oooo0?ooo`0Z0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool0<03oool00003
0?ooo`000000oooo00H0oooo00<000000?ooo`3oool0A@3oool200000080oooo00<000000?ooo`3o
ool0DP3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3o
ool0<@3oool00080oooo00<000000?ooo`3oool0103oool3000004@0oooo0P0000020?ooo`030000
003oool0oooo0580oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`0C0?ooo`030000
003oool0oooo0340oooo00030?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0@`3o
ool01@000000oooo0?ooo`3oool0000005@0oooo00<000000?ooo`3oool0:P3oool00`000000oooo
0?ooo`0C0?ooo`030000003oool0oooo0380oooo00040?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool0@P3oool01@000000oooo0?ooo`3oool0000005D0oooo00<000000?ooo`3oool0
:P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo03<0oooo00001@3oool000000?oo
o`3oool0000000@0oooo00<000000?ooo`3oool0@@3oool01@000000oooo0?ooo`3oool0000005D0
oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo03@0
oooo00020?ooo`8000001@3oool00`000000oooo0?ooo`100?ooo`050000003oool0oooo0?ooo`00
0000EP3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3o
ool0=@3oool000T0oooo00<000000?ooo`3oool0?`3oool200000080oooo00<000000?ooo`3oool0
E03oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0
=P3oool000T0oooo00<000000?ooo`3oool0?P3oool200000080oooo00<000000?ooo`3oool0E03o
ool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0=P3o
ool000T0oooo00<000000?ooo`3oool0?@3oool2000000<0oooo00<000000?ooo`3oool0E03oool0
0`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0=`3oool0
00T0oooo00<000000?ooo`3oool0?03oool2000000<0oooo00<000000?ooo`3oool0E03oool00`00
0000oooo0?ooo`0Z0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0>03oool000T0
oooo00<000000?ooo`3oool0>`3oool2000000<0oooo00<000000?ooo`3oool0E@3oool00`000000
oooo0?ooo`0Z0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0>@3oool000T0oooo
00<000000?ooo`3oool0>`3oool01@000000oooo0?ooo`3oool0000005L0oooo00<000000?ooo`3o
ool0:P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo03X0oooo00090?ooo`030000
003oool0oooo03X0oooo00D000000?ooo`3oool0oooo0000001G0?ooo`030000003oool0oooo02/0
oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0k0?ooo`002@3oool00`000000oooo
0?ooo`0i0?ooo`050000003oool0oooo0?ooo`000000F03oool00`000000oooo0?ooo`0Z0?ooo`03
0000003oool0oooo0100oooo00<000000?ooo`3oool0>`3oool000T0oooo00<000000?ooo`3oool0
>03oool01@000000oooo0?ooo`3oool0000005P0oooo00<000000?ooo`3oool0:P3oool00`000000
oooo0?ooo`0@0?ooo`030000003oool0oooo03`0oooo00090?ooo`030000003oool0oooo03L0oooo
0P0000020?ooo`030000003oool0oooo05H0oooo00<000000?ooo`3oool0:`3oool00`000000oooo
0?ooo`0?0?ooo`030000003oool0oooo03d0oooo00090?ooo`800000>03oool010000000oooo0?oo
o`00001I0?ooo`030000003oool0oooo02X0oooo00<000000?ooo`3oool03`3oool00`000000oooo
0?ooo`0n0?ooo`002@3oool00`000000oooo0?ooo`0f0?ooo`040000003oool0oooo000005T0oooo
00<000000?ooo`3oool0:`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo03l0oooo
00090?ooo`030000003oool0oooo03D0oooo00@000000?ooo`3oool00000F@3oool00`000000oooo
0?ooo`0[0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0@03oool000T0oooo00<0
00000?ooo`3oool0=03oool2000000030?ooo`000000oooo05P0oooo00<000000?ooo`3oool0;03o
ool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo0400oooo00090?ooo`030000003oool0
oooo03<0oooo0P0000000`3oool000000?ooo`1I0?ooo`030000003oool0oooo02/0oooo00<00000
0?ooo`3oool03P3oool00`000000oooo0?ooo`110?ooo`002@3oool00`000000oooo0?ooo`0b0?oo
o`80000000<0oooo0000003oool0F@3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo
00d0oooo00<000000?ooo`3oool0@P3oool000T0oooo00<000000?ooo`3oool0<P3oool3000005X0
oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo04<0
oooo00090?ooo`030000003oool0oooo0340oooo0`00001K0?ooo`030000003oool0oooo02/0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`140?ooo`002@3oool00`000000oooo0?oo
o`0a0?ooo`030000003oool0000005X0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?oo
o`0<0?ooo`030000003oool0oooo04D0oooo00090?ooo`030000003oool0oooo0300oooo00<00000
0?ooo`000000FP3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo00d0oooo00<00000
0?ooo`3oool0A@3oool000T0oooo00<000000?ooo`3oool0;`3oool2000000030?ooo`000000oooo
05T0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo
04H0oooo00090?ooo`030000003oool0oooo02l0oooo00<000000?ooo`000000FP3oool00`000000
oooo0?ooo`0/0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0A`3oool000T0oooo
00<000000?ooo`3oool0;P3oool010000000oooo0?ooo`00001I0?ooo`030000003oool0oooo02d0
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`180?ooo`002@3oool00`000000oooo
0?ooo`0^0?ooo`030000003oool0000005T0oooo00<000000?ooo`3oool0;@3oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo04T0oooo00090?ooo`800000;P3oool00`000000oooo0000
001J0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?oo
o`1:0?ooo`002@3oool00`000000oooo0?ooo`0/0?ooo`040000003oool0oooo000005T0oooo00<0
00000?ooo`3oool0;@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo04X0oooo0009
0?ooo`030000003oool0oooo02/0oooo0P0000000`3oool000000?ooo`1H0?ooo`030000003oool0
oooo02d0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`1;0?ooo`002@3oool00`00
0000oooo0?ooo`0[0?ooo`040000003oool0oooo000005T0oooo00<000000?ooo`3oool0;@3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo04`0oooo00090?ooo`030000003oool0oooo
02X0oooo0P0000000`3oool000000?ooo`1H0?ooo`030000003oool0oooo02d0oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`1=0?ooo`002@3oool00`000000oooo0?ooo`0Y0?ooo`80
00000P3oool00`000000oooo0?ooo`1F0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3o
ool02@3oool00`000000oooo0?ooo`1>0?ooo`002@3oool00`000000oooo0?ooo`0X0?ooo`800000
0P3oool00`000000oooo0?ooo`1G0?ooo`030000003oool0oooo02d0oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`1?0?ooo`002@3oool00`000000oooo0?ooo`0X0?ooo`80000000<0
oooo0000003oool0F03oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo00X0oooo00<0
00000?ooo`3oool0C`3oool000T0oooo00<000000?ooo`3oool09`3oool200000080oooo00<00000
0?ooo`3oool0EP3oool00`000000oooo0?ooo`0^0?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool0D03oool000T0oooo00<000000?ooo`3oool09`3oool010000000oooo0?ooo`00001H
0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`1A
0?ooo`002@3oool00`000000oooo0?ooo`0V0?ooo`8000000P3oool00`000000oooo0?ooo`1F0?oo
o`030000003oool0oooo02h0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`1B0?oo
o`002@3oool00`000000oooo0?ooo`0V0?ooo`040000003oool0oooo000005P0oooo00<000000?oo
o`3oool0;P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo05<0oooo00090?ooo`03
0000003oool0oooo02H0oooo00<000000?ooo`000000F03oool00`000000oooo0?ooo`0_0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool0E03oool000T0oooo00<000000?ooo`3oool0
9@3oool010000000oooo0?ooo`00001G0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3o
ool0203oool00`000000oooo0?ooo`1D0?ooo`002@3oool00`000000oooo0?ooo`0T0?ooo`800000
00<0oooo0000003oool0E`3oool00`000000oooo0?ooo`0_0?ooo`030000003oool0oooo00L0oooo
00<000000?ooo`3oool0E@3oool000T0oooo0P00000U0?ooo`040000003oool0oooo000005L0oooo
00<000000?ooo`3oool0;`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo05H0oooo
00090?ooo`030000003oool0oooo02<0oooo0P0000000`3oool000000?ooo`1F0?ooo`030000003o
ool0oooo02l0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`1G0?ooo`002@3oool0
0`000000oooo0?ooo`0S0?ooo`80000000<0oooo0000003oool0EP3oool00`000000oooo0?ooo`0_
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0F03oool000T0oooo00<000000?oo
o`3oool08P3oool2000000030?ooo`000000oooo05H0oooo00<000000?ooo`3oool0;`3oool00`00
0000oooo0?ooo`060?ooo`030000003oool0oooo05T0oooo00090?ooo`030000003oool0oooo0280
oooo0P0000000`3oool000000?ooo`1E0?ooo`030000003oool0oooo0300oooo00<000000?ooo`3o
ool01P3oool00`000000oooo0?ooo`1I0?ooo`002@3oool00`000000oooo0?ooo`0Q0?ooo`040000
003oool00000000005H0oooo00<000000?ooo`3oool0<03oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo05X0oooo00090?ooo`030000003oool0oooo0240oooo0P0000000`3oool00000
0?ooo`1E0?ooo`030000003oool0oooo0300oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`1K0?ooo`002@3oool00`000000oooo0?ooo`0Q0?ooo`80000000<0oooo0000003oool0E03o
ool00`000000oooo0?ooo`0`0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0G03o
ool000T0oooo00<000000?ooo`3oool08@3oool00`000000oooo0000001E0?ooo`030000003oool0
oooo0300oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`1M0?ooo`002@3oool00`00
0000oooo0?ooo`0P0?ooo`80000000<0oooo0000003oool0D`3oool00`000000oooo0?ooo`0a0?oo
o`030000003oool0oooo00@0oooo00<000000?ooo`3oool0GP3oool000T0oooo00<000000?ooo`3o
ool0803oool3000005D0oooo00<000000?ooo`3oool0<03oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo05h0oooo00090?ooo`030000003oool0oooo0200oooo00<000000?ooo`000000
E03oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
G`3oool000T0oooo00<000000?ooo`3oool07`3oool2000000030?ooo`000000oooo0580oooo00<0
00000?ooo`3oool0<@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0600oooo0009
0?ooo`030000003oool0oooo01l0oooo0`00001C0?ooo`030000003oool0oooo0380oooo00<00000
0?ooo`3oool00`3oool00`000000oooo0?ooo`1Q0?ooo`002@3oool200000200oooo00<000000?oo
o`000000D`3oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo00<0oooo00<000000?oo
o`3oool0HP3oool000T0oooo00<000000?ooo`3oool07`3oool00`000000oooo0000001B0?ooo`03
0000003oool0oooo0380oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`1S0?ooo`00
2@3oool00`000000oooo0?ooo`0N0?ooo`030000003oool000000580oooo00<000000?ooo`3oool0
<P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo06<0oooo00090?ooo`030000003o
ool0oooo01h0oooo00<000000?ooo`000000D@3oool00`000000oooo0?ooo`0b0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool0I03oool000T0oooo00<000000?ooo`3oool07P3oool0
0`000000oooo0000001A0?ooo`030000003oool0oooo0380oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`1U0?ooo`002@3oool00`000000oooo0?ooo`0M0?ooo`<00000D@3oool00`00
0000oooo0?ooo`0b0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0IP3oool000T0
oooo00<000000?ooo`3oool07@3oool00`000000oooo0000001@0?ooo`030000003oool0oooo03<0
oooo00D000000?ooo`3oool0oooo0000001Y0?ooo`002@3oool00`000000oooo0?ooo`0M0?ooo`03
0000003oool0000004l0oooo00<000000?ooo`3oool0<`3oool01@000000oooo0?ooo`3oool00000
06X0oooo00090?ooo`030000003oool0oooo01`0oooo0`00001@0?ooo`030000003oool0oooo03<0
oooo00D000000?ooo`3oool0oooo0000001Z0?ooo`002@3oool00`000000oooo0?ooo`0L0?ooo`<0
0000C`3oool00`000000oooo0?ooo`0c0?ooo`050000003oool0oooo0?ooo`000000J`3oool000T0
oooo00<000000?ooo`3oool0703oool200000500oooo00<000000?ooo`3oool0<P3oool01@000000
oooo0?ooo`3oool0000006`0oooo00090?ooo`030000003oool0oooo01`0oooo0P00001?0?ooo`03
0000003oool0oooo03<0oooo00@000000?ooo`3oool00000K@3oool000T0oooo00<000000?ooo`3o
ool06`3oool3000004h0oooo00<000000?ooo`3oool0<`3oool010000000oooo0?ooo`00001^0?oo
o`000P3oool2000000D0oooo00<000000?ooo`3oool06`3oool2000004h0oooo00<000000?ooo`3o
ool0=03oool00`000000oooo0000001_0?ooo`0000D0oooo0000003oool0oooo000000040?ooo`03
0000003oool0oooo01/0oooo0P00001=0?ooo`030000003oool0oooo03@0oooo00@000000?ooo`3o
ool00000K`3oool000@0oooo00<000000?ooo`3oool00P3oool3000001/0oooo0P00001<0?ooo`03
0000003oool0oooo03D0oooo00<000000?ooo`000000L03oool00080oooo0P0000050?ooo`030000
003oool0oooo01/0oooo0P0000230?ooo`030000003oool000000740oooo00040?ooo`030000003o
ool0oooo0080oooo00<000000?ooo`3oool06P3oool3000008<0oooo0P00001b0?ooo`0000D0oooo
0000003oool0oooo000000040?ooo`030000003oool0oooo01X0oooo0P0000230?ooo`800000L`3o
ool00080oooo0P0000050?ooo`030000003oool0oooo01X0oooo0P0000220?ooo`800000M03oool0
00T0oooo00<000000?ooo`3oool06P3oool200000880oooo00<000000?ooo`3oool0L`3oool000T0
oooo00<000000?ooo`3oool06P3oool200000840oooo0P00001e0?ooo`002@3oool00`000000oooo
0?ooo`0I0?ooo`<00000P@3oool00`000000oooo0?ooo`1d0?ooo`002@3oool00`000000oooo0?oo
o`0I0?ooo`<00000P03oool00`000000oooo0?ooo`1e0?ooo`002@3oool00`000000oooo0?ooo`0I
0?ooo`030000003oool0000007l0oooo0P00001g0?ooo`002@3oool00`000000oooo0?ooo`0I0?oo
o`800000O`3oool2000007P0oooo00090?ooo`030000003oool0oooo01T0oooo0P00001n0?ooo`80
0000N@3oool000T0oooo00<000000?ooo`3oool06@3oool2000007d0oooo0P00001j0?ooo`002@3o
ool00`000000oooo0?ooo`0I0?ooo`800000O03oool2000007/0oooo00090?ooo`8000006P3oool2
000007`0oooo00<000000?ooo`3oool0NP3oool000T0oooo00<000000?ooo`3oool0603oool00`00
0000oooo0000001k0?ooo`030000003oool0oooo07/0oooo00090?ooo`030000003oool0oooo01P0
oooo00<000000?ooo`000000NP3oool2000007d0oooo00090?ooo`030000003oool0oooo01P0oooo
0P00001j0?ooo`800000OP3oool000T0oooo00<000000?ooo`3oool0603oool2000007T0oooo0P00
001o0?ooo`002@3oool00`000000oooo0?ooo`0H0?ooo`800000N03oool200000800oooo00090?oo
o`030000003oool0oooo01P0oooo0P00001g0?ooo`800000P@3oool000T0oooo00<000000?ooo`3o
ool0603oool2000007L0oooo00<000000?ooo`3oool0P03oool000T0oooo00<000000?ooo`3oool0
603oool2000007H0oooo00<000000?ooo`3oool0P@3oool000T0oooo00<000000?ooo`3oool0603o
ool2000007D0oooo00<000000?ooo`3oool0PP3oool000T0oooo00<000000?ooo`3oool0603oool2
000007@0oooo00<000000?ooo`3oool0P`3oool000T0oooo00<000000?ooo`3oool0603oool00`00
0000oooo0?ooo`1b0?ooo`030000003oool0oooo08@0oooo00090?ooo`030000003oool0oooo01P0
oooo00<000000?ooo`3oool0L@3oool2000008H0oooo00090?ooo`030000003oool0oooo01P0oooo
00<000000?ooo`3oool0L03oool2000008L0oooo00090?ooo`030000003oool0oooo01P0oooo00<0
00000?ooo`3oool0L03oool00`000000oooo0?ooo`260?ooo`002@3oool2000001T0oooo00<00000
0?ooo`3oool0K`3oool00`000000oooo0?ooo`270?ooo`002@3oool00`000000oooo0?ooo`0H0?oo
o`030000003oool0oooo06h0oooo00<000000?ooo`3oool0R03oool000T0oooo00<000000?ooo`3o
ool0603oool00`000000oooo0?ooo`1]0?ooo`030000003oool0oooo08T0oooo00090?ooo`030000
003oool0oooo01P0oooo00<000000?ooo`3oool0K03oool00`000000oooo0?ooo`2:0?ooo`002@3o
ool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo06/0oooo00<000000?ooo`3oool0R`3o
ool000T0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`1Z0?ooo`030000003oool0
oooo08`0oooo00090?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0J@3oool00`00
0000oooo0?ooo`2=0?ooo`002@3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo06T0
oooo00<000000?ooo`3oool0S@3oool000T0oooo00<000000?ooo`3oool0603oool00`000000oooo
0?ooo`1X0?ooo`030000003oool0oooo08h0oooo00090?ooo`030000003oool0oooo01P0oooo00<0
00000?ooo`3oool0I`3oool00`000000oooo0?ooo`2?0?ooo`002@3oool00`000000oooo0?ooo`0H
0?ooo`030000003oool0oooo06H0oooo00<000000?ooo`3oool0T03oool000T0oooo00<000000?oo
o`3oool0603oool00`000000oooo0?ooo`1U0?ooo`030000003oool0oooo0940oooo00090?ooo`03
0000003oool0oooo01P0oooo00<000000?ooo`3oool0I03oool00`000000oooo0?ooo`2B0?ooo`00
2@3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo06<0oooo00<000000?ooo`3oool0
T`3oool000T0oooo0P00000I0?ooo`030000003oool0oooo0680oooo0P00002E0?ooo`002@3oool0
0`000000oooo0?ooo`0H0?ooo`030000003oool0oooo0640oooo0P00002F0?ooo`002@3oool00`00
0000oooo0?ooo`0H0?ooo`030000003oool0oooo0600oooo0P00002G0?ooo`002@3oool00`000000
oooo0?ooo`0H0?ooo`030000003oool0oooo0600oooo00<000000?ooo`3oool0UP3oool000T0oooo
00<000000?ooo`3oool0603oool00`000000oooo0?ooo`1O0?ooo`030000003oool0oooo09L0oooo
00090?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0GP3oool00`000000oooo0?oo
o`2H0?ooo`002@3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo05d0oooo00<00000
0?ooo`3oool0V@3oool000T0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`1L0?oo
o`800000V`3oool000T0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`1K0?ooo`80
0000W03oool000T0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`1J0?ooo`800000
W@3oool000T0oooo00<000000?ooo`3oool0603oool2000005X0oooo0P00002N0?ooo`002@3oool0
0`000000oooo0?ooo`0H0?ooo`800000F@3oool2000009l0oooo00090?ooo`030000003oool0oooo
01P0oooo0P00001H0?ooo`800000X03oool000T0oooo00<000000?ooo`3oool0603oool2000005P0
oooo00<000000?ooo`3oool0W`3oool000T0oooo0P00000I0?ooo`800000E`3oool00`000000oooo
0?ooo`2P0?ooo`002@3oool00`000000oooo0?ooo`0H0?ooo`800000EP3oool200000:80oooo0009
0?ooo`030000003oool0oooo01P0oooo0P00001E0?ooo`800000X`3oool000T0oooo00<000000?oo
o`3oool0603oool2000005@0oooo0P00002T0?ooo`002@3oool00`000000oooo0?ooo`0H0?ooo`80
0000D`3oool200000:D0oooo00090?ooo`030000003oool0oooo01P0oooo0P00001B0?ooo`800000
YP3oool000T0oooo00<000000?ooo`3oool0603oool200000540oooo0P00002W0?ooo`002@3oool0
0`000000oooo0?ooo`0I0?ooo`030000003oool0oooo04h0oooo0P00002X0?ooo`002@3oool00`00
0000oooo0?ooo`0I0?ooo`030000003oool0oooo04d0oooo00<000000?ooo`000000Z03oool000T0
oooo00<000000?ooo`3oool06@3oool2000004h0oooo0P00002Y0?ooo`002@3oool00`000000oooo
0?ooo`0I0?ooo`800000C@3oool200000:X0oooo00090?ooo`030000003oool0oooo01T0oooo0P00
001<0?ooo`800000Z`3oool000T0oooo00<000000?ooo`3oool06@3oool2000004/0oooo0P00002/
0?ooo`000`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01T0oooo0P00001:0?oo
o`800000[@3oool000<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0I0?ooo`80
0000B@3oool200000:h0oooo0005000000@0oooo0`00000I0?ooo`800000B03oool00`000000oooo
0000002^0?ooo`0000@000000?ooo`3oool000001@3oool00`000000oooo0?ooo`0I0?ooo`800000
A`3oool00`000000oooo0000002_0?ooo`0000@0oooo0000003oool000001@3oool00`000000oooo
0?ooo`0I0?ooo`800000AP3oool00`000000oooo0000002`0?ooo`000P3oool2000000D0oooo00<0
00000?ooo`3oool06@3oool2000004D0oooo00<000000?ooo`000000/@3oool000<0oooo00<00000
0?ooo`3oool00`3oool00`000000oooo0?ooo`0I0?ooo`800000A@3oool200000;80oooo00090?oo
o`030000003oool0oooo01X0oooo00<000000?ooo`3oool0@P3oool200000;<0oooo00090?ooo`03
0000003oool0oooo01X0oooo00<000000?ooo`3oool0@@3oool200000;@0oooo00090?ooo`030000
003oool0oooo01X0oooo00<000000?ooo`3oool0@03oool200000;D0oooo00090?ooo`030000003o
ool0oooo01X0oooo00<000000?ooo`3oool0?`3oool00`000000oooo0000002e0?ooo`002@3oool0
0`000000oooo0?ooo`0J0?ooo`800000?`3oool00`000000oooo0000002f0?ooo`002@3oool00`00
0000oooo0?ooo`0J0?ooo`800000?P3oool00`000000oooo0000002g0?ooo`002@3oool00`000000
oooo0?ooo`0J0?ooo`800000?@3oool00`000000oooo0000002h0?ooo`002@3oool00`000000oooo
0?ooo`0J0?ooo`800000?03oool00`000000oooo0000002i0?ooo`002@3oool00`000000oooo0?oo
o`0J0?ooo`800000>`3oool00`000000oooo0000002j0?ooo`002@3oool2000001`0oooo00<00000
0?ooo`3oool0>@3oool200000;/0oooo00090?ooo`030000003oool0oooo01/0oooo00<000000?oo
o`3oool0>03oool200000;`0oooo00090?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3o
ool0=`3oool200000;d0oooo00090?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool0
=P3oool200000;h0oooo00090?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool0=@3o
ool200000;l0oooo00090?ooo`030000003oool0oooo01/0oooo0P00000e0?ooo`800000`03oool0
00T0oooo00<000000?ooo`3oool06`3oool2000003@0oooo0P0000310?ooo`002@3oool00`000000
oooo0?ooo`0L0?ooo`030000003oool0oooo0340oooo0P0000320?ooo`002@3oool00`000000oooo
0?ooo`0L0?ooo`030000003oool0oooo0300oooo0P0000330?ooo`002@3oool00`000000oooo0?oo
o`0L0?ooo`030000003oool0oooo02l0oooo0P0000340?ooo`002@3oool00`000000oooo0?ooo`0L
0?ooo`800000<03oool00`000000oooo0?ooo`330?ooo`002@3oool00`000000oooo0?ooo`0L0?oo
o`800000;`3oool00`000000oooo0?ooo`340?ooo`002@3oool00`000000oooo0?ooo`0L0?ooo`80
0000;P3oool00`000000oooo0?ooo`350?ooo`002@3oool00`000000oooo0?ooo`0M0?ooo`030000
003oool0oooo02/0oooo00<000000?ooo`3oool0aP3oool000T0oooo00<000000?ooo`3oool07@3o
ool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo0<L0oooo00090?ooo`8000007P3oool2
000002X0oooo00<000000?ooo`3oool0b03oool000T0oooo00<000000?ooo`3oool07@3oool20000
02P0oooo0P00003;0?ooo`002@3oool00`000000oooo0?ooo`0M0?ooo`8000009`3oool200000<`0
oooo00090?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool0903oool200000<d0oooo
00090?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool08`3oool200000<h0oooo0009
0?ooo`030000003oool0oooo01h0oooo0P00000S0?ooo`030000003oool000000<h0oooo00090?oo
o`030000003oool0oooo01h0oooo0P00000R0?ooo`030000003oool000000<l0oooo00090?ooo`03
0000003oool0oooo01l0oooo00<000000?ooo`3oool07`3oool00`000000oooo0000003@0?ooo`00
2@3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo01h0oooo00<000000?ooo`000000
d@3oool000T0oooo00<000000?ooo`3oool07`3oool2000001h0oooo00<000000?ooo`000000dP3o
ool000T0oooo00<000000?ooo`3oool07`3oool2000001d0oooo00<000000?ooo`000000d`3oool0
00T0oooo00<000000?ooo`3oool0803oool00`000000oooo0?ooo`0J0?ooo`030000003oool00000
0=@0oooo00090?ooo`030000003oool0oooo0200oooo00<000000?ooo`3oool06@3oool300000=D0
oooo00090?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool05`3oool200000=L0oooo
00090?ooo`8000008P3oool00`000000oooo0?ooo`0F0?ooo`800000f03oool000T0oooo00<00000
0?ooo`3oool08P3oool00`000000oooo0?ooo`0D0?ooo`800000f@3oool000T0oooo00<000000?oo
o`3oool08P3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo0=T0oooo00090?ooo`03
0000003oool0oooo02<0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`3J0?ooo`00
2@3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo00l0oooo0P00003M0?ooo`002@3o
ool00`000000oooo0?ooo`0T0?ooo`030000003oool0oooo00d0oooo0P00003N0?ooo`002@3oool0
0`000000oooo0?ooo`0U0?ooo`030000003oool0oooo00X0oooo0`00003O0?ooo`002@3oool00`00
0000oooo0?ooo`0V0?ooo`030000003oool0oooo00L0oooo0`00003Q0?ooo`002@3oool00`000000
oooo0?ooo`0W0?ooo`8000001@3oool300000><0oooo00090?ooo`030000003oool0oooo02P0oooo
1P00003V0?ooo`002@3oool00`000000oooo0?ooo`3o0?oooaD0oooo00090?ooo`030000003oool0
oooo0?l0oooo5@3oool000T0oooo00<000000?ooo`3oool0o`3ooolE0?ooo`002@3oool00`000000
oooo0?ooo`3o0?oooaD0oooo00090?ooo`030000003oool0oooo0?l0oooo5@3oool000T0oooo0P00
003o0?oooaH0oooo00090?ooo`030000003oool0oooo0?l0oooo5@3oool00?l0oooo8@3oool00001
\
\>"],
  ImageRangeCache->{{{0, 287}, {296.375, 0}} -> {-0.132235, 0.736685, \
0.0138409, 0.0138409}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Method comparison", "Section",
  CellTags->"s:3"],

Cell[TextData[{
  "Sometimes you may be interested to find out what methods are being used in \
",
  StyleBox["NDSolve", "MR"],
  "."
}], "Text"],

Cell["\<\
Here you can see the coefficients of the default 2(1) embedded \
pair.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`EmbeddedExplicitRungeKuttaCoefficients[2, Infinity]\)], "Input",\

  CellLabel->"In[13]:="],

Cell[BoxData[
    \({{{1}, {1\/2, 1\/2}}, {1\/2, 1\/2, 0}, {1, 1}, {\(-\(1\/2\)\), 
        2\/3, \(-\(1\/6\)\)}}\)], "Output",
  CellLabel->"Out[13]="]
}, Open  ]],

Cell["\<\
Another issue is that you may want to compare some of the different \
methods to see how they perform for a specific problem.\
\>", "Text"],

Cell[CellGroupData[{

Cell["Utilities", "Subsection"],

Cell[TextData[{
  "We will make use of a utility function ",
  StyleBox["CompareMethods", "MR"],
  " for comparing various methods. Some useful ",
  StyleBox["NDSolve", "MR"],
  " features of this function for comparing methods are:"
}], "Text"],

Cell[TextData[{
  "The option ",
  StyleBox["EvaluationMonitor", "MR"],
  ", which is used to count the number of function evaluations."
}], "BulletedList"],

Cell[TextData[{
  "The option ",
  StyleBox["StepMonitor", "MR"],
  ", which is used to count the number of accepted and rejected integration \
steps."
}], "BulletedList"],

Cell[TextData[{
  "This displays the results of the method comparison using a ",
  StyleBox["GridBox", "MR"],
  "."
}], "MathCaption"],

Cell[BoxData[
    \(TabulateResults[labels_List, names_List, 
        data_List] \
:= \[IndentingNewLine]DisplayForm[\[IndentingNewLine]FrameBox[\
\[IndentingNewLine]GridBox[\[IndentingNewLine]Apply[{labels, ##} &, 
                MapThread[
                  Prepend, {data, 
                    names}]], \[IndentingNewLine]RowLines \[Rule] True, 
              ColumnLines \[Rule] 
                True\[IndentingNewLine]]\[IndentingNewLine]]\
\[IndentingNewLine]]\  /; \ SameQ[Length[names], \ Length[data]]; \)], "Input",\

  CellLabel->"In[14]:="]
}, Open  ]],

Cell[CellGroupData[{

Cell["Reference solution", "Subsection"],

Cell["\<\
A number of examples for comparing numerical methods in the \
literature rely on the fact that a closed-form solution is available, which \
is obviously quite limiting.\
\>", "Text"],

Cell[TextData[{
  "In ",
  StyleBox["NDSolve", "MR"],
  " it is possible to get very accurate approximations using \
arbitrary-precision adaptive step size, adaptive order methods based on ",
  StyleBox["Extrapolation", "MR"],
  "."
}], "Text"],

Cell[TextData[{
  "The following reference solution is computed with a method that switches \
between a pair of ",
  StyleBox["Extrapolation", "MR"],
  " methods, depending on whether the problem appears to be stiff."
}], "MathCaption"],

Cell[BoxData[{
    \(sol\  = \ 
      NDSolve[system, Method \[Rule] StiffnessSwitching, 
        WorkingPrecision \[Rule] 
          32]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(refsol\  = \ First[FinalSolutions[system, sol]]; \)}], "Input",
  CellLabel->"In[15]:="]
}, Open  ]],

Cell[CellGroupData[{

Cell["Automatic order selection", "Subsection"],

Cell[TextData[{
  "When you select ",
  StyleBox["DifferenceOrder\[Rule]Automatic", "MR"],
  ", the code will automatically attempt to choose the optimal order method \
for the integration."
}], "Text"],

Cell[TextData[{
  "Two algorithms have been implemented for this purpose and are described \
within ",
  StyleBox[ButtonBox["SymplecticPartitionedRungeKutta",
    ButtonData:>{"SPRK.nb", None},
    ButtonStyle->"Hyperlink"], "MR"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell["Example 1", "Subsubsection"],

Cell["\<\
Here is an example that compares built-in methods of various \
orders, together with the method that is selected automatically.\
\>", "Text"],

Cell[TextData[{
  "This selects the order of the methods to choose between and makes a list \
of method options to pass to ",
  StyleBox["NDSolve", "MR"],
  "."
}], "MathCaption"],

Cell[BoxData[{
    \(orders\  = \ 
      Join[Range[2, 
          9], {Automatic}]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(methods\  = \ 
      Table[{"\<ExplicitRungeKutta\>", 
          DifferenceOrder \[Rule] 
            Part[orders, i], \ "\<StiffnessTest\>" \[Rule] False}, {i, \ 
          Length[orders]}]; \)}], "Input",
  CellLabel->"In[17]:="],

Cell["\<\
Compute the number of integration steps, function evaluations, and \
the endpoint global error.\
\>", "MathCaption"],

Cell[BoxData[
    \(data\  = \ CompareMethods[system, \ refsol, \ methods]; \)], "Input",
  CellLabel->"In[19]:="],

Cell["Display the results in a table.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(labels\  = \ {"\<Method\>", "\<Steps\>", "\<Cost\>", "\<Error\>"}; \
\[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(TabulateResults[labels, \ orders, \ data]\)}], "Input",
  CellLabel->"In[20]:="],

Cell[BoxData[
    TagBox[
      FrameBox[GridBox[{
            {"Method", "Steps", "Cost", "Error"},
            {"2", \({117471, 2}\), "234948", \(2.1377750458739797  e - 08\)},
            {"3", \({4012, 2}\), "12044", \(4.0884006791142014  e - 08\)},
            {"4", \({889, 8}\), "3590", \(1.1084035023145589  e - 08\)},
            {"5", \({180, 29}\), "1465", \(6.7007454584988579  e - 09\)},
            {"6", \({276, 26}\), "2418", \(2.54244770167438  e - 10\)},
            {"7", \({158, 30}\), "1882", \(2.3712713459467238  e - 09\)},
            {"8", \({84, 25}\), "1419", \(2.2509498363752912  e - 08\)},
            {"9", \({86, 23}\), "1746", \(4.0519889665802111  e - 09\)},
            {"Automatic", \({86, 23}\), 
              "1747", \(4.0519889665802111  e - 09\)}
            },
          RowLines->True,
          ColumnLines->True]],
      DisplayForm]], "Output",
  CellLabel->"Out[21]//DisplayForm="]
}, Open  ]],

Cell["\<\
The default method has order nine, which is close to the optimal \
order of eight in this example. One function evaluation is needed during the \
initialization phase to determine the order.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Example 2", "Subsubsection"],

Cell["\<\
A limitation of the previous example is that it did not take into \
account the accuracy of the solution obtained by each method, so that it did \
not give a fair reflection of the cost.\
\>", "Text"],

Cell["\<\
Rather than taking a single tolerance to compare methods, it is \
preferable to use a range of tolerances.\
\>", "Text"],

Cell[TextData[{
  "The following example compares various ",
  StyleBox["ExplicitRungeKutta", "MR"],
  " methods of different orders using a variety of tolerances."
}], "Text"],

Cell[TextData[{
  "This selects the order of the methods to choose between and makes a list \
of method options to pass to ",
  StyleBox["NDSolve", "MR"],
  "."
}], "MathCaption"],

Cell[BoxData[{
    \(orders = 
      Join[Range[5, 
          9], {Automatic}]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(methods = 
      Table[{"\<ExplicitRungeKutta\>", 
          DifferenceOrder \[Rule] 
            Part[orders, i], "\<StiffnessTest\>" \[Rule] False}, {i, 
          Length[orders]}]; \)}], "Input",
  CellLabel->"In[21]:="],

Cell[TextData[{
  "The data comparing accuracy and work is computed using ",
  StyleBox["CompareMethods", "MR"],
  " for a range of tolerances."
}], "MathCaption"],

Cell[BoxData[
    \(data = 
      Table[Map[Rest, 
          CompareMethods[system, refsol, methods, AccuracyGoal \[Rule] tol, 
            PrecisionGoal \[Rule] tol]], {tol, 3, 12}]; \)], "Input",
  CellLabel->"In[24]:="],

Cell["This takes the logarithm of the work and error.", "MathCaption"],

Cell[BoxData[
    \(lgdata = 
      Transpose[
        Map[{Log[10. , Part[#1, 1]], Log[10. , Part[#1, 2]]} &, 
          data, {2}]]; \)], "Input",
  CellLabel->"In[25]:="],

Cell["\<\
The work-error comparison data for the various methods is displayed \
in the following plot, where the global error is displayed on the vertical \
axis and the number of function evaluations on the horizontal axis. The \
default order selected (displayed in red) can be seen to increase as the \
tolerances are increased.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(MultipleListPlot[lgdata, PlotJoined \[Rule] True, Axes \[Rule] False, 
      Frame \[Rule] True, PlotLegend \[Rule] orders, 
      PlotStyle \[Rule] {{Black}, {Black}, {Black}, {Black}, {Black}, {Red}}, 
      FrameTicks \[Rule] {LogScale[2, 4, 3], LogScale[\(-12\), \(-2\)], None, 
          None}]\)], "Input",
  CellLabel->"In[32]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .80794 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.408545 0.384735 0.403972 0.384735 [
[ 0 0 0 0 ]
[ 1 .80794 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
% Start of sub-graphic
p
0.0238095 0.0192368 0.79328 0.788707 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-1.42528 0.599629 0.735879 0.0519276 [
[.37361 -0.0125 -14 -9.8125 ]
[.37361 -0.0125 14 0 ]
[.97324 -0.0125 -17 -9.8125 ]
[.97324 -0.0125 17 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .11275 -52.25 -6.46875 ]
[-0.0125 .11275 0 6.46875 ]
[-0.0125 .2166 -52.25 -6.46875 ]
[-0.0125 .2166 0 6.46875 ]
[-0.0125 .32046 -48 -6.46875 ]
[-0.0125 .32046 0 6.46875 ]
[-0.0125 .42431 -48 -6.46875 ]
[-0.0125 .42431 0 6.46875 ]
[-0.0125 .52817 -40 -4.90625 ]
[-0.0125 .52817 0 4.90625 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ -0.0005 -0.0005 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.37361 0 m
.37361 .00625 L
s
gsave
.37361 -0.0125 -75 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1000) show
87.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.97324 0 m
.97324 .00625 L
s
gsave
.97324 -0.0125 -78 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10000) show
93.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.06008 0 m
.06008 .00375 L
s
.13499 0 m
.13499 .00375 L
s
.1931 0 m
.1931 .00375 L
s
.24058 0 m
.24058 .00375 L
s
.28073 0 m
.28073 .00375 L
s
.3155 0 m
.3155 .00375 L
s
.34617 0 m
.34617 .00375 L
s
.55412 0 m
.55412 .00375 L
s
.65971 0 m
.65971 .00375 L
s
.73462 0 m
.73462 .00375 L
s
.79273 0 m
.79273 .00375 L
s
.84021 0 m
.84021 .00375 L
s
.88035 0 m
.88035 .00375 L
s
.91513 0 m
.91513 .00375 L
s
.9458 0 m
.9458 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .11275 m
.00625 .11275 L
s
gsave
-0.0125 .11275 -113.25 -10.4688 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.9375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 14.375 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 14.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(12) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .2166 m
.00625 .2166 L
s
gsave
-0.0125 .2166 -113.25 -10.4688 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.9375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 14.375 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 14.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .32046 m
.00625 .32046 L
s
gsave
-0.0125 .32046 -109 -10.4688 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.9375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 14.375 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 14.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(8) show
105.125 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .42431 m
.00625 .42431 L
s
gsave
-0.0125 .42431 -109 -10.4688 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.9375 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 14.375 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 14.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 14.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(6) show
105.125 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .52817 m
.00625 .52817 L
s
gsave
-0.0125 .52817 -101 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.0001) show
99.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .16879 m
.00375 .16879 L
s
0 .18346 m
.00375 .18346 L
s
0 .19227 m
.00375 .19227 L
s
0 .1986 m
.00375 .1986 L
s
0 .20353 m
.00375 .20353 L
s
0 .20757 m
.00375 .20757 L
s
0 .211 m
.00375 .211 L
s
0 .21398 m
.00375 .21398 L
s
0 .27264 m
.00375 .27264 L
s
0 .28731 m
.00375 .28731 L
s
0 .29613 m
.00375 .29613 L
s
0 .30245 m
.00375 .30245 L
s
0 .30738 m
.00375 .30738 L
s
0 .31143 m
.00375 .31143 L
s
0 .31486 m
.00375 .31486 L
s
0 .31783 m
.00375 .31783 L
s
0 .3765 m
.00375 .3765 L
s
0 .39117 m
.00375 .39117 L
s
0 .39998 m
.00375 .39998 L
s
0 .40631 m
.00375 .40631 L
s
0 .41124 m
.00375 .41124 L
s
0 .41528 m
.00375 .41528 L
s
0 .41871 m
.00375 .41871 L
s
0 .42169 m
.00375 .42169 L
s
0 .48035 m
.00375 .48035 L
s
0 .49502 m
.00375 .49502 L
s
0 .50384 m
.00375 .50384 L
s
0 .51016 m
.00375 .51016 L
s
0 .51509 m
.00375 .51509 L
s
0 .51914 m
.00375 .51914 L
s
0 .52257 m
.00375 .52257 L
s
0 .52554 m
.00375 .52554 L
s
0 .58421 m
.00375 .58421 L
s
0 .59888 m
.00375 .59888 L
s
0 .60769 m
.00375 .60769 L
s
0 .61402 m
.00375 .61402 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
0 .61803 m
1 .61803 L
s
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.001 w
.02381 .58159 m
.1017 .60332 L
.18138 .48551 L
.27886 .45103 L
.3762 .37048 L
.47429 .31384 L
.56966 .27081 L
.67258 .22447 L
.78609 .17303 L
.90358 .11901 L
s
.0001 w
.02381 .58159 -1.875 0 Mabsadd m
.02381 .58159 0 2.5 Mabsadd L
.02381 .58159 1.875 0 Mabsadd L
.02381 .58159 0 -2.5 Mabsadd L
.02381 .58159 -1.875 0 Mabsadd L closepath F
.1017 .60332 -1.875 0 Mabsadd m
.1017 .60332 0 2.5 Mabsadd L
.1017 .60332 1.875 0 Mabsadd L
.1017 .60332 0 -2.5 Mabsadd L
.1017 .60332 -1.875 0 Mabsadd L closepath F
.18138 .48551 -1.875 0 Mabsadd m
.18138 .48551 0 2.5 Mabsadd L
.18138 .48551 1.875 0 Mabsadd L
.18138 .48551 0 -2.5 Mabsadd L
.18138 .48551 -1.875 0 Mabsadd L closepath F
.27886 .45103 -1.875 0 Mabsadd m
.27886 .45103 0 2.5 Mabsadd L
.27886 .45103 1.875 0 Mabsadd L
.27886 .45103 0 -2.5 Mabsadd L
.27886 .45103 -1.875 0 Mabsadd L closepath F
.3762 .37048 -1.875 0 Mabsadd m
.3762 .37048 0 2.5 Mabsadd L
.3762 .37048 1.875 0 Mabsadd L
.3762 .37048 0 -2.5 Mabsadd L
.3762 .37048 -1.875 0 Mabsadd L closepath F
.47429 .31384 -1.875 0 Mabsadd m
.47429 .31384 0 2.5 Mabsadd L
.47429 .31384 1.875 0 Mabsadd L
.47429 .31384 0 -2.5 Mabsadd L
.47429 .31384 -1.875 0 Mabsadd L closepath F
.56966 .27081 -1.875 0 Mabsadd m
.56966 .27081 0 2.5 Mabsadd L
.56966 .27081 1.875 0 Mabsadd L
.56966 .27081 0 -2.5 Mabsadd L
.56966 .27081 -1.875 0 Mabsadd L closepath F
.67258 .22447 -1.875 0 Mabsadd m
.67258 .22447 0 2.5 Mabsadd L
.67258 .22447 1.875 0 Mabsadd L
.67258 .22447 0 -2.5 Mabsadd L
.67258 .22447 -1.875 0 Mabsadd L closepath F
.78609 .17303 -1.875 0 Mabsadd m
.78609 .17303 0 2.5 Mabsadd L
.78609 .17303 1.875 0 Mabsadd L
.78609 .17303 0 -2.5 Mabsadd L
.78609 .17303 -1.875 0 Mabsadd L closepath F
.90358 .11901 -1.875 0 Mabsadd m
.90358 .11901 0 2.5 Mabsadd L
.90358 .11901 1.875 0 Mabsadd L
.90358 .11901 0 -2.5 Mabsadd L
.90358 .11901 -1.875 0 Mabsadd L closepath F
.001 w
.19621 .51133 m
.27998 .43661 L
.35471 .39006 L
.44153 .32184 L
.5223 .2627 L
.60268 .24177 L
.68937 .17315 L
.78178 .11441 L
.87834 .06606 L
.97619 .01472 L
s
.0001 w
.19621 .51133 0 2.5 Mabsadd m
.19621 .51133 -0.56518 .7779 Mabsadd L
.19621 .51133 -2.37764 .77254 Mabsadd L
.19621 .51133 -0.91448 -0.29713 Mabsadd L
.19621 .51133 -1.46946 -2.02254 Mabsadd L
.19621 .51133 0 -0.96154 Mabsadd L
.19621 .51133 1.46946 -2.02254 Mabsadd L
.19621 .51133 .91448 -0.29713 Mabsadd L
.19621 .51133 2.37764 .77254 Mabsadd L
.19621 .51133 .56518 .7779 Mabsadd L
.19621 .51133 0 2.5 Mabsadd L closepath F
.27998 .43661 0 2.5 Mabsadd m
.27998 .43661 -0.56518 .7779 Mabsadd L
.27998 .43661 -2.37764 .77254 Mabsadd L
.27998 .43661 -0.91448 -0.29713 Mabsadd L
.27998 .43661 -1.46946 -2.02254 Mabsadd L
.27998 .43661 0 -0.96154 Mabsadd L
.27998 .43661 1.46946 -2.02254 Mabsadd L
.27998 .43661 .91448 -0.29713 Mabsadd L
.27998 .43661 2.37764 .77254 Mabsadd L
.27998 .43661 .56518 .7779 Mabsadd L
.27998 .43661 0 2.5 Mabsadd L closepath F
.35471 .39006 0 2.5 Mabsadd m
.35471 .39006 -0.56518 .7779 Mabsadd L
.35471 .39006 -2.37764 .77254 Mabsadd L
.35471 .39006 -0.91448 -0.29713 Mabsadd L
.35471 .39006 -1.46946 -2.02254 Mabsadd L
.35471 .39006 0 -0.96154 Mabsadd L
.35471 .39006 1.46946 -2.02254 Mabsadd L
.35471 .39006 .91448 -0.29713 Mabsadd L
.35471 .39006 2.37764 .77254 Mabsadd L
.35471 .39006 .56518 .7779 Mabsadd L
.35471 .39006 0 2.5 Mabsadd L closepath F
.44153 .32184 0 2.5 Mabsadd m
.44153 .32184 -0.56518 .7779 Mabsadd L
.44153 .32184 -2.37764 .77254 Mabsadd L
.44153 .32184 -0.91448 -0.29713 Mabsadd L
.44153 .32184 -1.46946 -2.02254 Mabsadd L
.44153 .32184 0 -0.96154 Mabsadd L
.44153 .32184 1.46946 -2.02254 Mabsadd L
.44153 .32184 .91448 -0.29713 Mabsadd L
.44153 .32184 2.37764 .77254 Mabsadd L
.44153 .32184 .56518 .7779 Mabsadd L
.44153 .32184 0 2.5 Mabsadd L closepath F
.5223 .2627 0 2.5 Mabsadd m
.5223 .2627 -0.56518 .7779 Mabsadd L
.5223 .2627 -2.37764 .77254 Mabsadd L
.5223 .2627 -0.91448 -0.29713 Mabsadd L
.5223 .2627 -1.46946 -2.02254 Mabsadd L
.5223 .2627 0 -0.96154 Mabsadd L
.5223 .2627 1.46946 -2.02254 Mabsadd L
.5223 .2627 .91448 -0.29713 Mabsadd L
.5223 .2627 2.37764 .77254 Mabsadd L
.5223 .2627 .56518 .7779 Mabsadd L
.5223 .2627 0 2.5 Mabsadd L closepath F
.60268 .24177 0 2.5 Mabsadd m
.60268 .24177 -0.56518 .7779 Mabsadd L
.60268 .24177 -2.37764 .77254 Mabsadd L
.60268 .24177 -0.91448 -0.29713 Mabsadd L
.60268 .24177 -1.46946 -2.02254 Mabsadd L
.60268 .24177 0 -0.96154 Mabsadd L
.60268 .24177 1.46946 -2.02254 Mabsadd L
.60268 .24177 .91448 -0.29713 Mabsadd L
.60268 .24177 2.37764 .77254 Mabsadd L
.60268 .24177 .56518 .7779 Mabsadd L
.60268 .24177 0 2.5 Mabsadd L closepath F
.68937 .17315 0 2.5 Mabsadd m
.68937 .17315 -0.56518 .7779 Mabsadd L
.68937 .17315 -2.37764 .77254 Mabsadd L
.68937 .17315 -0.91448 -0.29713 Mabsadd L
.68937 .17315 -1.46946 -2.02254 Mabsadd L
.68937 .17315 0 -0.96154 Mabsadd L
.68937 .17315 1.46946 -2.02254 Mabsadd L
.68937 .17315 .91448 -0.29713 Mabsadd L
.68937 .17315 2.37764 .77254 Mabsadd L
.68937 .17315 .56518 .7779 Mabsadd L
.68937 .17315 0 2.5 Mabsadd L closepath F
.78178 .11441 0 2.5 Mabsadd m
.78178 .11441 -0.56518 .7779 Mabsadd L
.78178 .11441 -2.37764 .77254 Mabsadd L
.78178 .11441 -0.91448 -0.29713 Mabsadd L
.78178 .11441 -1.46946 -2.02254 Mabsadd L
.78178 .11441 0 -0.96154 Mabsadd L
.78178 .11441 1.46946 -2.02254 Mabsadd L
.78178 .11441 .91448 -0.29713 Mabsadd L
.78178 .11441 2.37764 .77254 Mabsadd L
.78178 .11441 .56518 .7779 Mabsadd L
.78178 .11441 0 2.5 Mabsadd L closepath F
.87834 .06606 0 2.5 Mabsadd m
.87834 .06606 -0.56518 .7779 Mabsadd L
.87834 .06606 -2.37764 .77254 Mabsadd L
.87834 .06606 -0.91448 -0.29713 Mabsadd L
.87834 .06606 -1.46946 -2.02254 Mabsadd L
.87834 .06606 0 -0.96154 Mabsadd L
.87834 .06606 1.46946 -2.02254 Mabsadd L
.87834 .06606 .91448 -0.29713 Mabsadd L
.87834 .06606 2.37764 .77254 Mabsadd L
.87834 .06606 .56518 .7779 Mabsadd L
.87834 .06606 0 2.5 Mabsadd L closepath F
.97619 .01472 0 2.5 Mabsadd m
.97619 .01472 -0.56518 .7779 Mabsadd L
.97619 .01472 -2.37764 .77254 Mabsadd L
.97619 .01472 -0.91448 -0.29713 Mabsadd L
.97619 .01472 -1.46946 -2.02254 Mabsadd L
.97619 .01472 0 -0.96154 Mabsadd L
.97619 .01472 1.46946 -2.02254 Mabsadd L
.97619 .01472 .91448 -0.29713 Mabsadd L
.97619 .01472 2.37764 .77254 Mabsadd L
.97619 .01472 .56518 .7779 Mabsadd L
.97619 .01472 0 2.5 Mabsadd L closepath F
.001 w
.1889 .54437 m
.26223 .45638 L
.31288 .40346 L
.39171 .35609 L
.47429 .32683 L
.53268 .29223 L
.60824 .24679 L
.67991 .1922 L
.75236 .14812 L
.83237 .09141 L
s
.0001 w
.1889 .54437 -2.5 2.5 Mabsadd m
.1889 .54437 2.5 2.5 Mabsadd L
.1889 .54437 2.5 -2.5 Mabsadd L
.1889 .54437 -2.5 -2.5 Mabsadd L
.1889 .54437 -2.5 2.5 Mabsadd L closepath F
.26223 .45638 -2.5 2.5 Mabsadd m
.26223 .45638 2.5 2.5 Mabsadd L
.26223 .45638 2.5 -2.5 Mabsadd L
.26223 .45638 -2.5 -2.5 Mabsadd L
.26223 .45638 -2.5 2.5 Mabsadd L closepath F
.31288 .40346 -2.5 2.5 Mabsadd m
.31288 .40346 2.5 2.5 Mabsadd L
.31288 .40346 2.5 -2.5 Mabsadd L
.31288 .40346 -2.5 -2.5 Mabsadd L
.31288 .40346 -2.5 2.5 Mabsadd L closepath F
.39171 .35609 -2.5 2.5 Mabsadd m
.39171 .35609 2.5 2.5 Mabsadd L
.39171 .35609 2.5 -2.5 Mabsadd L
.39171 .35609 -2.5 -2.5 Mabsadd L
.39171 .35609 -2.5 2.5 Mabsadd L closepath F
.47429 .32683 -2.5 2.5 Mabsadd m
.47429 .32683 2.5 2.5 Mabsadd L
.47429 .32683 2.5 -2.5 Mabsadd L
.47429 .32683 -2.5 -2.5 Mabsadd L
.47429 .32683 -2.5 2.5 Mabsadd L closepath F
.53268 .29223 -2.5 2.5 Mabsadd m
.53268 .29223 2.5 2.5 Mabsadd L
.53268 .29223 2.5 -2.5 Mabsadd L
.53268 .29223 -2.5 -2.5 Mabsadd L
.53268 .29223 -2.5 2.5 Mabsadd L closepath F
.60824 .24679 -2.5 2.5 Mabsadd m
.60824 .24679 2.5 2.5 Mabsadd L
.60824 .24679 2.5 -2.5 Mabsadd L
.60824 .24679 -2.5 -2.5 Mabsadd L
.60824 .24679 -2.5 2.5 Mabsadd L closepath F
.67991 .1922 -2.5 2.5 Mabsadd m
.67991 .1922 2.5 2.5 Mabsadd L
.67991 .1922 2.5 -2.5 Mabsadd L
.67991 .1922 -2.5 -2.5 Mabsadd L
.67991 .1922 -2.5 2.5 Mabsadd L closepath F
.75236 .14812 -2.5 2.5 Mabsadd m
.75236 .14812 2.5 2.5 Mabsadd L
.75236 .14812 2.5 -2.5 Mabsadd L
.75236 .14812 -2.5 -2.5 Mabsadd L
.75236 .14812 -2.5 2.5 Mabsadd L closepath F
.83237 .09141 -2.5 2.5 Mabsadd m
.83237 .09141 2.5 2.5 Mabsadd L
.83237 .09141 2.5 -2.5 Mabsadd L
.83237 .09141 -2.5 -2.5 Mabsadd L
.83237 .09141 -2.5 2.5 Mabsadd L closepath F
.001 w
.15443 .57606 m
.21697 .48375 L
.29165 .47814 L
.32225 .4229 L
.39389 .36232 L
.45253 .34313 L
.52984 .27956 L
.57882 .23674 L
.64507 .18419 L
.70457 .13102 L
s
.0001 w
.15443 .57606 0 2.5 Mabsadd m
.15443 .57606 -2.165 -1.25 Mabsadd L
.15443 .57606 2.165 -1.25 Mabsadd L
.15443 .57606 0 2.5 Mabsadd L closepath F
.21697 .48375 0 2.5 Mabsadd m
.21697 .48375 -2.165 -1.25 Mabsadd L
.21697 .48375 2.165 -1.25 Mabsadd L
.21697 .48375 0 2.5 Mabsadd L closepath F
.29165 .47814 0 2.5 Mabsadd m
.29165 .47814 -2.165 -1.25 Mabsadd L
.29165 .47814 2.165 -1.25 Mabsadd L
.29165 .47814 0 2.5 Mabsadd L closepath F
.32225 .4229 0 2.5 Mabsadd m
.32225 .4229 -2.165 -1.25 Mabsadd L
.32225 .4229 2.165 -1.25 Mabsadd L
.32225 .4229 0 2.5 Mabsadd L closepath F
.39389 .36232 0 2.5 Mabsadd m
.39389 .36232 -2.165 -1.25 Mabsadd L
.39389 .36232 2.165 -1.25 Mabsadd L
.39389 .36232 0 2.5 Mabsadd L closepath F
.45253 .34313 0 2.5 Mabsadd m
.45253 .34313 -2.165 -1.25 Mabsadd L
.45253 .34313 2.165 -1.25 Mabsadd L
.45253 .34313 0 2.5 Mabsadd L closepath F
.52984 .27956 0 2.5 Mabsadd m
.52984 .27956 -2.165 -1.25 Mabsadd L
.52984 .27956 2.165 -1.25 Mabsadd L
.52984 .27956 0 2.5 Mabsadd L closepath F
.57882 .23674 0 2.5 Mabsadd m
.57882 .23674 -2.165 -1.25 Mabsadd L
.57882 .23674 2.165 -1.25 Mabsadd L
.57882 .23674 0 2.5 Mabsadd L closepath F
.64507 .18419 0 2.5 Mabsadd m
.64507 .18419 -2.165 -1.25 Mabsadd L
.64507 .18419 2.165 -1.25 Mabsadd L
.64507 .18419 0 2.5 Mabsadd L closepath F
.70457 .13102 0 2.5 Mabsadd m
.70457 .13102 -2.165 -1.25 Mabsadd L
.70457 .13102 2.165 -1.25 Mabsadd L
.70457 .13102 0 2.5 Mabsadd L closepath F
.001 w
.28295 .563 m
.31615 .458 L
.38029 .4448 L
.40727 .35253 L
.46309 .36058 L
.51875 .32081 L
.582 .27261 L
.63591 .20987 L
.68437 .17029 L
.73683 .11744 L
s
.01 w
.28295 .563 Mdot
.31615 .458 Mdot
.38029 .4448 Mdot
.40727 .35253 Mdot
.46309 .36058 Mdot
.51875 .32081 Mdot
.582 .27261 Mdot
.63591 .20987 Mdot
.68437 .17029 Mdot
.73683 .11744 Mdot
1 0 0 r
.001 w
.07607 .5998 m
.19466 .55457 L
.31321 .40346 L
.39196 .35609 L
.46327 .36058 L
.5189 .32081 L
.58211 .27261 L
.636 .20987 L
.68445 .17029 L
.73689 .11744 L
s
0 g
.0001 w
.07607 .5998 -1.875 0 Mabsadd m
.07607 .5998 0 2.5 Mabsadd L
.07607 .5998 1.875 0 Mabsadd L
.07607 .5998 0 -2.5 Mabsadd L
.07607 .5998 -1.875 0 Mabsadd L closepath F
.19466 .55457 -1.875 0 Mabsadd m
.19466 .55457 0 2.5 Mabsadd L
.19466 .55457 1.875 0 Mabsadd L
.19466 .55457 0 -2.5 Mabsadd L
.19466 .55457 -1.875 0 Mabsadd L closepath F
.31321 .40346 -1.875 0 Mabsadd m
.31321 .40346 0 2.5 Mabsadd L
.31321 .40346 1.875 0 Mabsadd L
.31321 .40346 0 -2.5 Mabsadd L
.31321 .40346 -1.875 0 Mabsadd L closepath F
.39196 .35609 -1.875 0 Mabsadd m
.39196 .35609 0 2.5 Mabsadd L
.39196 .35609 1.875 0 Mabsadd L
.39196 .35609 0 -2.5 Mabsadd L
.39196 .35609 -1.875 0 Mabsadd L closepath F
.46327 .36058 -1.875 0 Mabsadd m
.46327 .36058 0 2.5 Mabsadd L
.46327 .36058 1.875 0 Mabsadd L
.46327 .36058 0 -2.5 Mabsadd L
.46327 .36058 -1.875 0 Mabsadd L closepath F
.5189 .32081 -1.875 0 Mabsadd m
.5189 .32081 0 2.5 Mabsadd L
.5189 .32081 1.875 0 Mabsadd L
.5189 .32081 0 -2.5 Mabsadd L
.5189 .32081 -1.875 0 Mabsadd L closepath F
.58211 .27261 -1.875 0 Mabsadd m
.58211 .27261 0 2.5 Mabsadd L
.58211 .27261 1.875 0 Mabsadd L
.58211 .27261 0 -2.5 Mabsadd L
.58211 .27261 -1.875 0 Mabsadd L closepath F
.636 .20987 -1.875 0 Mabsadd m
.636 .20987 0 2.5 Mabsadd L
.636 .20987 1.875 0 Mabsadd L
.636 .20987 0 -2.5 Mabsadd L
.636 .20987 -1.875 0 Mabsadd L closepath F
.68445 .17029 -1.875 0 Mabsadd m
.68445 .17029 0 2.5 Mabsadd L
.68445 .17029 1.875 0 Mabsadd L
.68445 .17029 0 -2.5 Mabsadd L
.68445 .17029 -1.875 0 Mabsadd L closepath F
.73689 .11744 -1.875 0 Mabsadd m
.73689 .11744 0 2.5 Mabsadd L
.73689 .11744 1.875 0 Mabsadd L
.73689 .11744 0 -2.5 Mabsadd L
.73689 .11744 -1.875 0 Mabsadd L closepath F
MathSubEnd
P
% End of sub-graphic
0 g
.83175 .25008 m
.83175 .55787 L
.97619 .55787 L
.97619 .25008 L
F
1 g
.83175 .25008 m
.83175 .55787 L
.97619 .55787 L
.97619 .25008 L
F
0 g
.001 w
[ ] 0 setdash
.83175 .25008 m
.97619 .25008 L
.97619 .55787 L
.83175 .55787 L
.83175 .25008 L
s
% Start of sub-graphic
p
0.831754 0.250078 0.97619 0.557866 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0297619 0.297619 0.0297619 0.297619 [
[ 0 0 0 0 ]
[ 1 2.13095 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 2.13095 L
0 2.13095 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0535714 0.0535714 0.35119 0.35119 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.5 0.5 0.5 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
1 0 0 r
.01403 w
[ ] 0 setdash
0 .5 m
1 .5 L
s
0 g
.0014 w
.5 .5 -1.875 0 Mabsadd m
.5 .5 0 2.5 Mabsadd L
.5 .5 1.875 0 Mabsadd L
.5 .5 0 -2.5 Mabsadd L
.5 .5 -1.875 0 Mabsadd L closepath F
MathSubEnd
P
% End of sub-graphic
0 g
gsave
.38988 .20238 -61 -9.0625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18.125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.562 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Automatic) show
1.000 setlinewidth
grestore
% Start of sub-graphic
p
0.0535714 0.39881 0.35119 0.696429 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.5 0.5 0.5 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
0 g
.01403 w
[ ] 0 setdash
0 .5 m
1 .5 L
s
.14025 w
.5 .5 Mdot
MathSubEnd
P
% End of sub-graphic
0 g
[(9)] .38988 .54762 -1 0 Mshowa
% Start of sub-graphic
p
0.0535714 0.744048 0.35119 1.04167 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.5 0.5 0.5 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
0 g
.01403 w
[ ] 0 setdash
0 .5 m
1 .5 L
s
.0014 w
.5 .5 0 2.5 Mabsadd m
.5 .5 -2.165 -1.25 Mabsadd L
.5 .5 2.165 -1.25 Mabsadd L
.5 .5 0 2.5 Mabsadd L closepath F
MathSubEnd
P
% End of sub-graphic
0 g
[(8)] .38988 .89286 -1 0 Mshowa
% Start of sub-graphic
p
0.0535714 1.08929 0.35119 1.3869 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.5 0.5 0.5 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
0 g
.01403 w
[ ] 0 setdash
0 .5 m
1 .5 L
s
.0014 w
.5 .5 -2.5 2.5 Mabsadd m
.5 .5 2.5 2.5 Mabsadd L
.5 .5 2.5 -2.5 Mabsadd L
.5 .5 -2.5 -2.5 Mabsadd L
.5 .5 -2.5 2.5 Mabsadd L closepath F
MathSubEnd
P
% End of sub-graphic
0 g
[(7)] .38988 1.2381 -1 0 Mshowa
% Start of sub-graphic
p
0.0535714 1.43452 0.35119 1.73214 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.5 0.5 0.5 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
0 g
.01403 w
[ ] 0 setdash
0 .5 m
1 .5 L
s
.0014 w
.5 .5 0 2.5 Mabsadd m
.5 .5 -0.56518 .7779 Mabsadd L
.5 .5 -2.37764 .77254 Mabsadd L
.5 .5 -0.91448 -0.29713 Mabsadd L
.5 .5 -1.46946 -2.02254 Mabsadd L
.5 .5 0 -0.96154 Mabsadd L
.5 .5 1.46946 -2.02254 Mabsadd L
.5 .5 .91448 -0.29713 Mabsadd L
.5 .5 2.37764 .77254 Mabsadd L
.5 .5 .56518 .7779 Mabsadd L
.5 .5 0 2.5 Mabsadd L closepath F
MathSubEnd
P
% End of sub-graphic
0 g
[(6)] .38988 1.58333 -1 0 Mshowa
% Start of sub-graphic
p
0.0535714 1.77976 0.35119 2.07738 MathSubStart
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.5 0.5 0.5 [
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
0 g
.01403 w
[ ] 0 setdash
0 .5 m
1 .5 L
s
.0014 w
.5 .5 -1.875 0 Mabsadd m
.5 .5 0 2.5 Mabsadd L
.5 .5 1.875 0 Mabsadd L
.5 .5 0 -2.5 Mabsadd L
.5 .5 -1.875 0 Mabsadd L closepath F
MathSubEnd
P
% End of sub-graphic
0 g
[(5)] .38988 1.92857 -1 0 Mshowa
MathSubEnd
P
% End of sub-graphic
0 0 m
1 0 L
1 .80794 L
0 .80794 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[32]:=",
  ImageSize->{288, 232.625},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000j2000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00J`3oool4000000<0oooo0P0000040?ooo`800000103oool2000004@0oooo100000030?oo
o`800000103oool2000000@0oooo0P0000040?ooo`800000@@3oool006d0oooo00D000000?ooo`3o
ool0oooo000000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3o
ool00`000000oooo0?ooo`130?ooo`050000003oool0oooo0?ooo`0000000P3oool010000000oooo
0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool0
0`000000oooo0?ooo`0n0?ooo`00K@3oool01@000000oooo0?ooo`3oool000000080oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo04<0oooo
00D000000?ooo`3oool0oooo000000020?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo03h0oooo001]
0?ooo`050000003oool0oooo0?ooo`0000000P3oool010000000oooo0?ooo`0000020?ooo`040000
003oool0oooo00000080oooo00<000000?ooo`3oool0@`3oool01@000000oooo0?ooo`3oool00000
0080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003o
ool0oooo00000080oooo00<000000?ooo`3oool0?P3oool006d0oooo00D000000?ooo`3oool0oooo
000000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool00`00
0000oooo0?ooo`130?ooo`050000003oool0oooo0?ooo`0000000P3oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool00`000000
oooo0?ooo`0n0?ooo`00J`3oool3000000<0oooo00@000000?ooo`3oool000000P3oool010000000
oooo0?ooo`0000020?ooo`040000003oool0oooo000004<0oooo0`0000030?ooo`040000003oool0
oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`04
0000003oool0oooo00000400oooo001]0?ooo`030000003oool0oooo0080oooo0P0000040?ooo`80
0000103oool2000004H0oooo00<000000?ooo`3oool00P3oool2000000@0oooo0P0000040?ooo`80
0000103oool2000000l0oooo:`0000070?ooo`00kP3oool00`000000oooo0?ooo`0W0?ooo`030000
003oool0oooo00D0oooo000m0?oooiX000005`3oool00`000000oooo0?ooo`0W0?ooo`030000003o
ool0oooo00D0oooo000m0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0203oool0
0`000000oooo0?ooo`060?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00D000000?ooo`3oool0oooo0000
000K0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool0203oool00`000000oooo0?oo
o`060?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool0oooo0P0000020?ooo`030000
003oool0oooo01D0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`050?ooo`00?@3o
ool00`000000oooo0?ooo`2B0?ooo`8000000P3oool00`000000oooo0?ooo`0E0?ooo`030000003o
ool0oooo02L0oooo00<000000?ooo`3oool01@3oool003d0oooo00<000000?ooo`3oool0T03oool5
000000030?ooo`000000oooo01H0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`05
0?ooo`00?@3oool00`000000oooo0?ooo`2>0?ooo`8000001P3oool00`000000oooo0?ooo`0E0?oo
o`030000003oool0oooo00l0oooo0P0000020?ooo`8000000P3oool2000000030?ooo`000000oooo
0080oooo0P0000030?ooo`8000000P3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000
oooo0?ooo`2<0?ooo`800000203oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool0203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
000000<0oooo00H000000?ooo`3oool000000?ooo`0000020?ooo`030000003oool0000000L0oooo
000m0?ooo`030000003oool0oooo08X0oooo0P00000:0?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3o0000103o0003000000H0o`000`3oool400000080oooo00@000000?ooo`3oool00000
0`3oool00`000000oooo0?ooo`020?ooo`060000003oool0oooo0000003oool000001`3oool003d0
oooo00<000000?ooo`3oool0R03oool2000000`0oooo00<000000?ooo`3oool05@3oool00`000000
oooo0?ooo`050?ooo`8000002@3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
000000<0oooo00<000000?ooo`3oool00P3oool01P000000oooo0?ooo`000000oooo000000L0oooo
000m0?ooo`030000003oool0oooo08H0oooo0P00000>0?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`090?ooo`040000003oool000000?ooo`800000
00<0oooo0000000000000P3oool4000000<0oooo0P0000020?ooo`030000003oool0oooo00D0oooo
000m0?ooo`030000003oool0oooo08<0oooo0`00000@0?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool0403oool3000000X0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?oo
o`050?ooo`00?@3oool00`000000oooo0?ooo`230?ooo`030000003oool0oooo0100oooo00<00000
0?ooo`3oool05@3oool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo00L0oooo00<00000
0?ooo`3oool01@3oool003d0oooo00<000000?ooo`3oool0P@3oool500000100oooo00<000000?oo
o`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000m0?ooo`03
0000003oool0oooo07l0oooo0P0000020?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000m0?ooo`030000
003oool0oooo07X0oooo1@00000G0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0
9`3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`1j0?ooo`D000005`3o
ool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool01@3o
ool000T0oooo100000030?ooo`030000003oool0oooo00@0oooo00D000000?ooo`3oool0oooo0000
00030?ooo`@000000`3oool2000001D0oooo00<000000?ooo`3oool0NP3oool5000001L0oooo00<0
00000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000;
0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`0000001P3oool01@000000oooo0?ooo`3o
ool000000080oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`1h0?ooo`L000005`3o
ool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool01@3o
ool000/0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`050?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0B0?ooo`800000KP3oool2000000D0oooo1@00
00000`3oool0000000000003000000L0oooo0P00000>0?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool09`3oool00`000000oooo0?ooo`050?ooo`002`3oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0000000H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool0K03oool400000080oooo1P00000=0?ooo`@000003@3o
ool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00h0oooo0`00000F0?ooo`030000003o
ool0oooo00D0oooo000;0?ooo`030000003oool0oooo00T0oooo00D000000?ooo`3oool0oooo0000
00050?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0B0?ooo`030000
003oool0oooo06d0oooo0`0000030?ooo`<000003P3oool5000000d0oooo00<000000?ooo`3oool0
5@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool0
1@3oool000T0oooo0`00000C0?ooo`<000000`3oool010000000oooo0?ooo`0000060?ooo`L00000
1`3oool00`000000oooo0?ooo`1X0?ooo`<0000000@0oooo0?l0003oool000000P3oool200000003
0?ooo`000000oooo00`0oooo0P0000030?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3o
ool05@3oool00`000000oooo0000000=000000@0oooo00<000000?ooo`3oool04`3oool00`000000
oooo0?ooo`050?ooo`002`3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo0080oooo
0P0000030?ooo`<000000P3oool00`000000oooo0?ooo`02000000P0oooo00<000000?ooo`3oool0
J03oool2000000040?ooo`3o0000oooo0?ooo`H00000303oool2000001@0oooo00<000000?ooo`3o
ool05@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00L0oooo0`00000E0?ooo`03
0000003oool0oooo00D0oooo000`0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
1@3oool00`000000oooo0?ooo`1W0?ooo`040000003oool000000?l000<0oooo1@00000;0?ooo`80
00005P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00h0oooo00@000000?ooo`3o
ool000005@3oool00`000000oooo0?ooo`050?ooo`00<03oool00`000000oooo0?ooo`03000000L0
oooo00<000000?ooo`3oool0IP3oool010000000oooo0?ooo`3o00020?ooo`L000002@3oool20000
01P0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0>0?ooo`040000003oool0oooo
000001D0oooo00<000000?ooo`3oool01@3oool003d0oooo00<000000?ooo`3oool0I03oool20000
0080oooo00@0o`000?ooo`3oool000000P3oool5000000L0oooo0P00000J0?ooo`030000003oool0
oooo01D0oooo00<000000?ooo`3oool03`3oool2000001H0oooo00<000000?ooo`3oool01@3oool0
03d0oooo00<000000?ooo`3oool0H`3oool01P000000oooo000000000000o`000?ooo`8000000P3o
ool600000080oooo00<000000?ooo`3oool00P00000L0?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool09`3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`1R
0?ooo`030000003oool0000000@000000P3oool2000000L0oooo1000000N0?ooo`030000003oool0
oooo01D0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`00
0000oooo0?ooo`1Q0?ooo`040000003oool0oooo0?l000H00000203oool4000001l0oooo00<00000
0?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000m0?oo
o`030000003oool0oooo05l0oooo0`0000000`3oool0o`0000000004000000T0oooo1000000P0?oo
o`030000003oool0oooo01D0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`050?oo
o`00?@3oool00`000000oooo0?ooo`1N0?ooo`<0000000<0oooo0?l000000000100000080?ooo`80
0000903oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3o
ool01@3oool000T0oooo100000030?ooo`030000003oool0oooo00@0oooo00D000000?ooo`3oool0
oooo000000030?ooo`@000000`3oool2000001D0oooo00<000000?ooo`3oool0G@3oool00`000000
oooo0?ooo`020?l000H000001P3oool2000002H0oooo00<000000?ooo`3oool05@3oool00`000000
oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000;0?ooo`030000003oool0oooo00X0oooo
00<000000?ooo`0000001P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3o
ool04P3oool00`000000oooo0?ooo`1L0?ooo`<0000000@0o`000?ooo`000000oooo1@0000040?oo
o`800000:03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo00<000000?oo
o`3oool01@3oool000/0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`050?ooo`05
0000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo
05X0oooo1`0000000`3oool000000000000400000080oooo0P00000Z0?ooo`030000003oool0oooo
01D0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`050?ooo`002`3oool00`000000
oooo0?ooo`0:0?ooo`030000003oool0000000H0oooo00D000000?ooo`3oool0oooo000000020?oo
o`030000003oool0oooo0180oooo0P00001J0?ooo`040000003oool0oooo0?ooo`<0000000D0oooo
0000003oool0oooo000000030?ooo`800000;03oool00`000000oooo0?ooo`0E0?ooo`030000003o
ool0oooo00l0oooo0P00000F0?ooo`030000003oool0oooo00D0oooo000;0?ooo`030000003oool0
oooo00T0oooo00D000000?ooo`3oool0oooo000000050?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo05P0oooo00D000000?ooo`3oool0
oooo0?l0000400000080oooo1@00000^0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3o
ool0103oool4000000H0oooo00@000000?ooo`3oool000005@3oool00`000000oooo0?ooo`050?oo
o`002@3oool3000001<0oooo0`0000030?ooo`040000003oool0oooo000000H0oooo0`0000000`3o
ool00000000000080?ooo`030000003oool0oooo05H0oooo0P0000030?ooo`040000003o0000oooo
000000<0oooo1000000`0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`0000003@000000
1@3oool000000?ooo`3oool0000001D0oooo00<000000?ooo`3oool01@3oool000/0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`020?ooo`8000000`3oool300000080oooo00<00000
0?ooo`0000000P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo05@0oooo1`000000
0`3o0000oooo000000030?ooo`@00000<@3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0
oooo00D0oooo00<000000?ooo`3oool01`3oool2000001H0oooo00<000000?ooo`3oool01@3oool0
0300oooo00<000000?ooo`0000000P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
05@0oooo1P0000000`3o00000000000000020?ooo`800000=@3oool00`000000oooo0?ooo`0E0?oo
o`030000003oool0oooo00h0oooo00@000000?ooo`3oool000005@3oool00`000000oooo0?ooo`05
0?ooo`00<03oool00`000000oooo0?ooo`02000000P0oooo00<000000?ooo`3oool0D`3oool60000
00030?l000000000000000<00000=`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo
00h0oooo00@000000?ooo`3oool000005@3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`00
0000oooo0?ooo`1?0?ooo`@000001@3oool00`3o0000000000000002000003T0oooo00<000000?oo
o`3oool05@3oool00`000000oooo0?ooo`0?0?ooo`8000005P3oool00`000000oooo0?ooo`050?oo
o`00?@3oool00`000000oooo0?ooo`1<0?ooo`<000000P3oool01@000000oooo0?ooo`000000oooo
0080000000<0o`000000000000000P00000i0?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool09`3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`1:0?ooo`D0
000000@0oooo0000003oool0oooo2P00000i0?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool09`3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`1:0?ooo`80
00000`3oool00`000000oooo0?ooo`06000003h0oooo00<000000?ooo`3oool05@3oool00`000000
oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000m0?ooo`030000003oool0oooo04T0oooo
00@000000?ooo`3oool0oooo0`0000001@3oool00000000000000000o`0000800000?`3oool00`00
0000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool01@3oool003d0
oooo00<000000?ooo`3oool0B03oool010000000oooo0?ooo`3oool500000080oooo00<0o`000?oo
o`3oool0@03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo00<000000?oo
o`3oool01@3oool003d0oooo00<000000?ooo`3oool0AP3oool2000000<0oooo1P0000020?l004<0
oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0
oooo000>0?ooo`@000000`3oool00`000000oooo0?ooo`040?ooo`050000003oool0oooo0?ooo`00
00000P3oool4000000<0oooo0P00000A0?ooo`030000003oool0oooo04D0oooo00@000000?ooo`3o
ool0oooo1`0000000`3o0000oooo0?ooo`130?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool09`3oool00`000000oooo0?ooo`050?ooo`00403oool00`000000oooo0?ooo`0:0?ooo`03
0000003oool0000000D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00h0oooo00<000000?ooo`3oool0A03oool2000000030?ooo`000000000000H0000000<0o`000000
003oool0A03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00<0oooo1@0000070?oo
o`030000003oool0oooo01D0oooo00<000000?ooo`3oool01@3oool00100oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`040?ooo`050000003oool0oooo0?ooo`0000000P3oool00`00
0000oooo0?ooo`0>0?ooo`030000003oool0oooo0480oooo1@000000103oool000000000003oool2
00000080o`000P0000150?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool00`3oool5
000000L0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`050?ooo`00403oool00`00
0000oooo0?ooo`0:0?ooo`030000003oool0000000D0oooo00D000000?ooo`3oool0oooo00000002
0?ooo`030000003oool0oooo00h0oooo0P0000100?ooo`<0000000<0oooo0000000000000`000002
0?ooo`@00000B03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0000000d000000`3oool0
0`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00D0oooo000@0?ooo`030000003oool0oooo
00T0oooo00D000000?ooo`3oool0oooo000000040?ooo`050000003oool0oooo0?ooo`0000000P3o
ool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo03h0oooo2P0000030?ooo`<00000B03o
ool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00<0oooo1@0000080?ooo`030000003o
ool0oooo01@0oooo00<000000?ooo`3oool01@3oool000h0oooo0`00000B0?ooo`<000000`3oool0
10000000oooo0?ooo`0000070?ooo`<000001P3oool00`000000oooo0?ooo`0o0?ooo`80000000<0
oooo000000000000100000020?ooo`030?l0003oool0000004T0oooo00<000000?ooo`3oool05@3o
ool00`000000oooo0?ooo`030?ooo`D000002@3oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo00D0oooo000@0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool00P3oool2
000000<0oooo0`0000020?ooo`030000003oool0000000H0oooo00<000000?ooo`3oool0?@3oool5
000000030?ooo`000000000000<000000P3o001<0?ooo`030000003oool0oooo01D0oooo00<00000
0?ooo`3oool03P3oool010000000oooo0?ooo`00000E0?ooo`030000003oool0oooo00D0oooo000e
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0=P3oool5000000040?ooo`000000
000000000080oooo1P0000000`3o0000oooo0?ooo`1<0?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool03P3oool4000001D0oooo00<000000?ooo`3oool01@3oool003@0oooo0`000006
0?ooo`030000003oool0oooo03H0oooo2`0000001P3oool000000?ooo`000000oooo0?l004l0oooo
00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo
000m0?ooo`030000003oool0oooo03H0oooo1@0000040?l000<0000000@0oooo0000000000000000
D03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool0
1@3oool003d0oooo00<000000?ooo`3oool0=P3oool00`000000o`0000000002000000@0oooo0`3o
000400000500oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003o
ool0oooo00D0oooo000m0?ooo`030000003oool0oooo03D0oooo0P3o0004000000P0oooo0P00001A
0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`05
0?ooo`00?@3oool00`000000oooo0?ooo`0c0?ooo`80o`000`0000020?ooo`030000003oool0oooo
05T0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo
00D0oooo000m0?ooo`030000003oool0oooo0380oooo00<0o`000000000000000P0000030?ooo`03
0000003oool0oooo05T0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`03
0000003oool0oooo00D0oooo000m0?ooo`030000003oool0oooo0300oooo0P3o0003000000@0oooo
00<000000?ooo`3oool0FP3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo
00<000000?ooo`3oool01@3oool003d0oooo00<000000?ooo`3oool0:P3oool400000080o`0000<0
oooo0000000000000P0000040?ooo`030000003oool0oooo05X0oooo00<000000?ooo`3oool05@3o
ool00`000000oooo0?ooo`0?0?ooo`8000005P3oool00`000000oooo0?ooo`050?ooo`003P3oool4
000000<0oooo00<000000?ooo`3oool0103oool01@000000oooo0?ooo`3oool000000080oooo1000
00030?ooo`8000004@3oool00`000000oooo0?ooo`0Z0?ooo`D0000000<0oooo0000003oool00P00
00050?ooo`030000003oool0oooo05X0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?oo
o`040?ooo`<000001`3oool010000000oooo0?ooo`00000E0?ooo`030000003oool0oooo00D0oooo
000@0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`0000001@3oool01@000000oooo0?oo
o`3oool000000080oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0Z0?ooo`P00000
1P3oool00`000000oooo0?ooo`1K0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`000000
3@0000001@3oool000000?ooo`3oool0000001D0oooo00<000000?ooo`3oool01@3oool00100oooo
00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`040?ooo`050000003oool0oooo0?ooo`00
00000P3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo02X0oooo00<000000?l00000
00000`0000080?ooo`030000003oool0oooo05/0oooo00<000000?ooo`3oool05@3oool00`000000
oooo0?ooo`030?ooo`D000001P3oool3000001H0oooo00<000000?ooo`3oool01@3oool00100oooo
00<000000?ooo`3oool02P3oool00`000000oooo000000050?ooo`050000003oool0oooo0?ooo`00
00000P3oool00`000000oooo0?ooo`0>0?ooo`800000:`3oool00`000000o`0000000003000000P0
oooo00<000000?ooo`3oool0F`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0
oooo000@0?ooo`030000003oool0oooo00T0oooo00D000000?ooo`3oool0oooo000000040?ooo`05
0000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo
02L0oooo0`0000000`3o000000000?ooo`02000000T0oooo00<000000?ooo`3oool0F`3oool00`00
0000oooo0?ooo`0E0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool05P3oool00`00
0000oooo0?ooo`050?ooo`003P3oool300000180oooo0`0000030?ooo`040000003oool0oooo0000
00L0oooo0`0000060?ooo`030000003oool0oooo02L0oooo0P000000103o000000000?ooo`3oool2
000000P0oooo00<000000?ooo`3oool0G03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0
oooo00l0oooo0`00000E0?ooo`030000003oool0oooo00D0oooo000@0?ooo`030000003oool0oooo
0180oooo00<000000?ooo`3oool00P3oool2000000<0oooo0`0000001@3oool000000?ooo`3oool0
000000H0oooo00<000000?ooo`3oool09@3oool3000000050?l000000000oooo0?ooo`000000203o
ool3000005h0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003o
ool0oooo00D0oooo000c0?ooo`@000001P3oool00`000000oooo0?ooo`0R0?ooo`H0000000D0o`00
0?ooo`3oool0oooo000000030?ooo`D000000P3oool00`000000oooo0?ooo`1L0?ooo`030000003o
ool0oooo01D0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`050?ooo`00=03oool3
000000H0oooo00<000000?ooo`3oool08P3oool5000000060?l000000000oooo0?ooo`000000oooo
0`00001V0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool09`3oool00`000000oooo
0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`0Q0?ooo`D0000000<0o`000000003oool00P3o
ool00`000000oooo0000001X0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool09`3o
ool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`0O0?ooo`80000000H0oooo
00000000000000000?l0000000030?ooo`030000003oool0000006T0oooo00<000000?ooo`3oool0
5@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000m0?ooo`030000003o
ool0oooo01P0oooo00@000000?ooo`3oool0oooo0`0000020?ooo`<0000000@0o`00000000000000
oooo1@00001Y0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool09`3oool00`000000
oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`0H0?ooo`@0000000<0oooo000000000000
1@0000000`3o0000000000000004000000030?ooo`000000oooo06P0oooo00<000000?ooo`3oool0
5@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000m0?ooo`030000003o
ool0oooo01L0oooo0`0000030?ooo`<0000000@0oooo0000003oool0o`001@3oool010000000oooo
0?ooo`00001Y0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool03P3oool3000001H0
oooo00<000000?ooo`3oool01@3oool003d0oooo00<000000?ooo`3oool05`3oool2000000@0oooo
0P000000103oool000000?ooo`3o00080?ooo`030000003oool0oooo06P0oooo00<000000?ooo`3o
ool05@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`0G0?ooo`030000
003oool0oooo0080oooo0P0000020?ooo`030000003o0000oooo00P0oooo00<000000?ooo`3oool0
J03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0000000d00000103oool00`000000oooo
0?ooo`0C0?ooo`030000003oool0oooo00D0oooo000m0?ooo`030000003oool0oooo01H0oooo00@0
00000?ooo`3oool0oooo0`0000020?ooo`030000003o0000oooo00T0oooo00<000000?ooo`3oool0
J03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00D0oooo0P0000070?ooo`<00000
5P3oool00`000000oooo0?ooo`050?ooo`00?@3oool00`000000oooo0?ooo`0E0?ooo`030000003o
ool0oooo00D0000000@0oooo0000003oool0o`002@3oool00`000000oooo0?ooo`1Y0?ooo`030000
003oool0oooo01D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`060?ooo`030000
003oool0oooo01H0oooo00<000000?ooo`3oool01@3oool001L0oooo0P0000040?ooo`030000003o
ool0oooo00<0oooo0P0000040?ooo`800000103oool2000000<0oooo100000050?ooo`030000003o
ool0oooo01D0oooo00<000000?ooo`3oool00P3oool01P000000oooo0?ooo`000000oooo0?l000X0
oooo00<000000?ooo`3oool0J@3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`050?ooo`005P3oool010000000oooo
0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool0
10000000oooo0?ooo`0000040?ooo`030000003oool0oooo00@0oooo0P00000E0?ooo`030000003o
ool0oooo0080oooo00H000000?ooo`3oool000000?ooo`3o000;0?ooo`030000003oool0oooo06T0
oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0>0?ooo`@000005@3oool00`000000
oooo0?ooo`050?ooo`005P3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool04`3oool010000000oooo0?ooo`3oool6000000030?l0
003oool0oooo00T0oooo00<000000?ooo`3oool0JP3oool00`000000oooo0?ooo`0E0?ooo`030000
003oool0oooo02L0oooo00<000000?ooo`3oool01@3oool001H0oooo00@000000?ooo`3oool00000
203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool00000103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01<0oooo00@00000
0?ooo`3oool0oooo1@0000000`3oool0o`000?ooo`0:0?ooo`030000003oool0oooo06X0oooo00<0
00000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00D0oooo000F
0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo
0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`0B0?ooo`030000003oool0oooo0080oooo1@0000000`3o0000oooo0?ooo`0:0?oo
o`030000003oool0oooo06X0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0W0?oo
o`030000003oool0oooo00D0oooo000F0?ooo`040000003oool0oooo000000P0oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo
0`0000060?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool00`3oool5000000d0oooo
00<000000?ooo`3oool0JP3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo02L0oooo
00<000000?ooo`3oool01@3oool001L0oooo0P00000:0?ooo`800000103oool2000000@0oooo0P00
00050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?oo
o`020?ooo`040000003o0000o`000?l000800000303oool00`000000oooo0?ooo`1[0?ooo`030000
003oool0oooo01D0oooo:`0000070?ooo`00?@3oool00`000000oooo0?ooo`0@0?ooo`030000003o
ool0oooo0080oooo0P3o00040?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0J`3o
ool00`000000oooo0?ooo`170?ooo`00?@3oool010000000oooo00000000000>0?ooo`030000003o
ool0oooo00<0o`0000<000000?ooo`3oool0OP3oool00`000000oooo0?ooo`170?ooo`00?@3oool5
000000d0oooo0`3o00000`3oool00000000000210?ooo`030000003oool0oooo04L0oooo000m0?oo
o`030000003oool0000000D00000203oool20?l000D0oooo00<000000?ooo`3oool0O`3oool00`00
0000oooo0?ooo`170?ooo`00?@3oool010000000oooo0?ooo`0000040?ooo`@0000000D0oooo0?l0
003o0000o`00000000280?ooo`030000003oool0oooo04L0oooo000m0?ooo`030000003oool0oooo
00H0oooo200000280?ooo`030000003oool0oooo04L0oooo000m0?ooo`030000003oool0oooo00L0
oooo1`0000280?ooo`030000003oool0oooo04L0oooo000m0?ooo`030000003oool0oooo00P0oooo
00<000000?ooo`3oool00P0000290?ooo`030000003oool0oooo04L0oooo000m0?oooiX00000B@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {231.625, 0}} -> {-1.06331, -1.05, 0.00906634, \
0.00906634}, {{6.9375, 227.562}, {226.062, 5.5}} -> {1.71331, -23.4834, \
0.01086, 0.125405}, {{238.562, 280}, {159.875, 71.6875}} -> {-19.4712, \
-5.92542, 0.0811906, 0.0811906}, {{240.75, 253.062}, {157.625, 145.312}} -> \
{-40.107, -13.0204, 0.162438, 0.162438}, {{240.75, 253.062}, {143.312, 131}} -> \
{-40.107, -15.3453, 0.162438, 0.162438}, {{240.75, 253.062}, {129.062, \
116.75}} -> {-40.107, -17.6601, 0.162438, 0.162438}, {{240.75, 253.062}, \
{114.75, 102.438}} -> {-40.107, -19.985, 0.162438, 0.162438}, {{240.75, \
253.062}, {100.5, 88.1875}} -> {-40.107, -22.2997, 0.162438, 0.162438}, \
{{240.75, 253.062}, {86.1875, 73.875}} -> {-40.107, -24.6246, 0.162438, \
0.162438}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[32]="]
}, Open  ]],

Cell["\<\
The order selection algorithms are heuristic in that the optimal \
order may change through the integration but, as the above examples \
illustrate, a reasonable default choice is usually made.\
\>", "Text"],

Cell["\<\
Ideally, a selection of different problems should be used for \
benchmarking.\
\>", "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Coefficient plug-in", "Section",
  CellTags->"s:4"],

Cell[TextData[{
  "The implementation of ",
  StyleBox["ExplicitRungeKutta", "MR"],
  " provides a default method pair at each order."
}], "Text"],

Cell["\<\
Sometimes, however, it is convenient to use a different method, for \
example:\
\>", "Text"],

Cell["To replicate the results of someone else.", "BulletedList"],

Cell["\<\
To use a special-purpose method that works well for a specific \
problem.\
\>", "BulletedList"],

Cell["To experiment with a new method", "BulletedList"],

Cell[CellGroupData[{

Cell["The classical Runge-Kutta method", "Subsection"],

Cell[TextData[{
  "This shows how to define the coefficients of the classical explicit \
Runge-Kutta method of order four, approximated to precision ",
  StyleBox["p", "MR"],
  "."
}], "MathCaption"],

Cell["\<\
crkamat = {{1/2}, {0, 1/2}, {0, 0, 1}};
crkbvec = {1/6, 1/3, 1/3, 1/6};
crkcvec = {1/2, 1/2, 1};
ClassicalRungeKuttaCoefficients[4, p_] :=
  N[{crkamat,crkbvec, crkcvec}, p];\
\>", "Input",
  CellLabel->"In[20]:="],

Cell["\<\
The method has no embedded error estimate and hence there is no \
specification of the coefficient error vector. This means that the method is \
invoked with fixed step sizes.\
\>", "Text"],

Cell["Here is an example of the calling syntax.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[system, 
      Method \[Rule] {"\<ExplicitRungeKutta\>", 
          DifferenceOrder \[Rule] 4, "\<Coefficients\>" \[Rule] 
            ClassicalRungeKuttaCoefficients}, 
      StartingStepSize \[Rule] 1/10]\)], "Input",
  CellLabel->"In[24]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[24]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["ode23", "Subsection"],

Cell["\<\
This defines the coefficients for a 3(2) FSAL explicit Runge-Kutta \
pair.\
\>", "Text"],

Cell[TextData[{
  "The third-order formula is due to Ralston, and the embedded method was \
derived by Bogacki and Shampine [",
  ButtonBox["BS89a",
    ButtonData:>{"References.nb", "BS89a"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell["\<\
This defines function for computing the coefficients to a desired \
precision.\
\>", "MathCaption"],

Cell[BoxData[{
    \(BSamat = {{1/2}, {0, 3/4}, {2/9, 1/3, 4/9}}; \), "\n", 
    \(BSbvec = {2/9, 1/3, 4/9, 0}; \), "\n", 
    \(BScvec\  = \ {1/2, 3/4, 1}; \), "\[IndentingNewLine]", 
    \(BSevec = {\(-5\)/72, 1/12, 
        1/9, \(-1\)/8}; \n\), "\[IndentingNewLine]", 
    \(BSCoefficients[4, \ 
        p_] := \[IndentingNewLine]N[{BSamat, BSbvec, BScvec, BSevec}, \ 
        p]; \)}], "Input",
  CellLabel->"In[25]:="],

Cell[TextData[{
  "The method is used in the Texas Instruments TI-85 pocket calculator, \
Matlab and RKSUITE [",
  ButtonBox["S94",
    ButtonData:>{"References.nb", "S94"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell["\<\
Unfortunately it does not allow for the form of stiffness detection \
that has been chosen.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["A method of Fehlberg", "Subsection"],

Cell[TextData[{
  "This defines the coefficients for a 4(5) explicit Runge-Kutta pair of \
Fehlberg that was popular in the 1960s [",
  ButtonBox["F69",
    ButtonData:>{"References.nb", "F69"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell["\<\
The fourth order formula is used to propagate the solution and the \
fifth-order formula is used only for the purpose of error estimation.\
\>", \
"Text"],

Cell["\<\
This defines function for computing the coefficients to a desired \
precision.\
\>", "MathCaption"],

Cell[BoxData[{
    \(Fehlbergamat = {\[IndentingNewLine]{1/4}, \[IndentingNewLine]{3/32, 
          9/32}, \[IndentingNewLine]{1932/2197, \(-7200\)/2197, 
          7296/2197}, {439/216, \(-8\), 3680/513, \(-845\)/4104}, {\(-8\)/27, 
          2, \(-3544\)/2565, 1859/4104, \(-11\)/40}}; \), "\n", 
    \(Fehlbergbvec = {25/216, 0, 1408/2565, 2197/4104, \(-1\)/5, \ 
        0}; \), "\n", 
    \(Fehlbergcvec\  = \ {1/4, 3/8, 12/13, 1, 
        1/2}; \), "\[IndentingNewLine]", 
    \(Fehlbergevec = {\(-1\)/360, 0, 128/4275, 
        2197/75240, \(-1\)/50, \(-2\)/55}; \n\), "\[IndentingNewLine]", 
    \(FehlbergCoefficients[4, \ 
        p_] := \[IndentingNewLine]N[{Fehlbergamat, Fehlbergbvec, 
          Fehlbergcvec, Fehlbergevec}, \ p]; \)}], "Input",
  CellLabel->"In[30]:="],

Cell["\<\
In contrast to the classical Runge-Kutta method of order four, the \
coefficients include an additional entry that is used for error estimation.\
\
\>", "Text"],

Cell[TextData[{
  "The Fehlberg method is not a FSAL scheme since the coefficient matrix is \
not of the form (",
  
  CounterBox["NumberedEquation", "FSALButcherTable"],
  "); it is a six-stage scheme, but it requires six function evaluations per \
step because of the function evaluation that is required at the end of the \
step to construct the ",
  StyleBox["InterpolatingFunction", "MR"],
  "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["A DOrmand-PRInce method", "Subsection"],

Cell[TextData[{
  "Here is how to define a 5(4) pair of Dormand and Prince [",
  ButtonBox["DP80",
    ButtonData:>{"References.nb", "DP80"},
    ButtonStyle->"Hyperlink"],
  "]. This is currently the method used by ",
  StyleBox["ode45", "MR"],
  " in Matlab."
}], "Text"],

Cell["\<\
This defines function for computing the coefficients to a desired \
precision.\
\>", "MathCaption"],

Cell[BoxData[{
    \(DOPRIamat = {\[IndentingNewLine]{1/5}, \[IndentingNewLine]{3/40, 
          9/40}, \[IndentingNewLine]{44/45, \(-56\)/15, 
          32/9}, \[IndentingNewLine]{19372/6561, \(-25360\)/2187, 
          64448/6561, \(-212\)/729}, \[IndentingNewLine]{9017/3168, \(-355\)/
            33, 46732/5247, 
          49/176, \(-5103\)/18656}, \[IndentingNewLine]{35/384, 0, 500/1113, 
          125/192, \(-2187\)/6784, 11/84}}; \), "\n", 
    \(DOPRIbvec = {35/384, 0, 500/1113, 125/192, \(-2187\)/6784, 11/84, \ 
        0}; \), "\n", 
    \(DOPRIcvec\  = \ {1/5, 3/10, 4/5, 8/9, 1, \ 
        1}; \), "\[IndentingNewLine]", 
    \(DOPRIevec = {71/57600, 0, \(-71\)/16695, 71/1920, \(-17253\)/339200, 
        22/525, \(-1\)/40}; \[IndentingNewLine]\), "\n", 
    \(DOPRICoefficients[5, \ 
        p_] := \[IndentingNewLine]N[{DOPRIamat, DOPRIbvec, DOPRIcvec, 
          DOPRIevec}, \ p]; \)}], "Input",
  CellLabel->"In[35]:="],

Cell[TextData[{
  "The Dormand-Prince method is a FSAL scheme since the coefficient matrix is \
of the form (",
  
  CounterBox["NumberedEquation", "FSALButcherTable"],
  "); it is a seven-stage scheme, but effectively uses only six function \
evaluations."
}], "Text"],

Cell["\<\
Here is how the coefficients of Dormand and Princeton be used  in \
place of the built-in choice. Since the structure of the coefficients \
includes an error vector, the implementation is able to ascertain that \
adaptive step sizes can be computed.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[system, 
      Method \[Rule] {"\<ExplicitRungeKutta\>", 
          DifferenceOrder \[Rule] 5, "\<Coefficients\>" \[Rule] 
            DOPRICoefficients, \ "\<StiffnessTest\>" \[Rule] 
            False}]\)], "Input",
  CellLabel->"In[40]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[40]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Method comparison", "Subsection"],

Cell["\<\
Here we solve a system using several explicit Runge-Kutta \
pairs.\
\>", "Text"],

Cell[TextData[{
  "For the Fehlberg 4(5) pair, the option ",
  StyleBox["EmbeddedDifferenceOrder", "MR"],
  " is used to specify the order of the embedded method."
}], "MathCaption"],

Cell[BoxData[
    \(Fehlberg45\  = \ {"\<ExplicitRungeKutta\>", "\<Coefficients\>" \[Rule] 
          FehlbergCoefficients, 
        DifferenceOrder \[Rule] 4, "\<EmbeddedDifferenceOrder\>" \[Rule] 
          5, \ "\<StiffnessTest\>" \[Rule] False}; \)], "Input",
  CellLabel->"In[41]:="],

Cell["The Dormand and Prince 5(4) pair is defined as follows.", "MathCaption"],

Cell[BoxData[
    \(DOPRI54\  = {"\<ExplicitRungeKutta\>", "\<Coefficients\>" \[Rule] 
          DOPRICoefficients, 
        DifferenceOrder \[Rule] 5, \ "\<StiffnessTest\>" \[Rule] 
          False}; \)], "Input",
  CellLabel->"In[42]:="],

Cell["\<\
The 5(4) pair of Bogacki and Shampine is the default order-five \
method.\
\>", "MathCaption"],

Cell[BoxData[
    \(BS54\  = {"\<ExplicitRungeKutta\>", "\<Coefficients\>" \[Rule] \
"\<EmbeddedExplicitRungeKuttaCoefficients\>", 
        DifferenceOrder \[Rule] 5, \ "\<StiffnessTest\>" \[Rule] 
          False}; \)], "Input",
  CellLabel->"In[43]:="],

Cell["\<\
Put the methods and some descriptive names together in a \
list.\
\>", "MathCaption"],

Cell[BoxData[{
    \(names\  = \ {"\<Fehlberg 4(5)\>", \ "\<Dormand-Prince 5(4)\>", \
"\<Bogacki-Shampine 5(4)\>"}; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(methods\  = \ {Fehlberg45, DOPRI54, BS54}; \)}], "Input",
  CellLabel->"In[44]:="],

Cell["\<\
Compute the number of integration steps, function evaluations, and \
the endpoint global error.\
\>", "MathCaption"],

Cell[BoxData[
    \(data\  = \ CompareMethods[system, refsol, \ methods]; \)], "Input",
  CellLabel->"In[46]:="],

Cell["Display the results in a table.", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(labels\  = \ {"\<Method\>", "\<Steps\>", "\<Cost\>", "\<Error\>"}; \
\[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(TabulateResults[labels, \ names, \ data]\)}], "Input",
  CellLabel->"In[47]:="],

Cell[BoxData[
    TagBox[
      FrameBox[GridBox[{
            {"Method", "Steps", "Cost", "Error"},
            {\(Fehlberg\ 4 \((5)\)\), \({304, 21}\), 
              "1931", \(1.8793891332080648  e - 07\)},
            {\(Dormand - Prince\ 5 \((4)\)\), \({278, 22}\), 
              "1802", \(2.0743949002977047  e - 08\)},
            {\(Bogacki - Shampine\ 5 \((4)\)\), \({180, 29}\), 
              "1465", \(6.7007454584988579  e - 09\)}
            },
          RowLines->True,
          ColumnLines->True]],
      DisplayForm]], "Output",
  CellLabel->"Out[48]//DisplayForm="]
}, Open  ]],

Cell["\<\
The default method was the least expensive and provided the most \
accurate solution.\
\>", "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Method plug-in", "Section",
  CellTags->"s:5"],

Cell["\<\
This shows how to implement the classical explicit Runge-Kutta \
method of order four using the method plug-in environment.\
\>", "Text"],

Cell["\<\
This definition is optional since the method in fact has no data. \
However, any expression can be stored inside the data object. For example, \
the coefficients could be approximated here to avoid coercion from rational \
to floating-point numbers at each integration step.\
\>", "MathCaption"],

Cell["\<\
ClassicalRungeKutta /:
  NDSolve`InitializeMethod[ClassicalRungeKutta, __] := ClassicalRungeKutta[];\
\
\>", "Input",
  CellLabel->"In[49]:="],

Cell["\<\
The actual method implementation is written using a stepping \
procedure.\
\>", "MathCaption"],

Cell["\<\
ClassicalRungeKutta[___][\"Step\"[f_, t_, h_, y_, yp_]]:=
  Block[{deltay, k1, k2, k3, k4},
    k1 = yp;
    k2 = f[t + 1/2 h, y + 1/2 h k1];
    k3 = f[t + 1/2 h, y + 1/2 h k2];
    k4 = f[t + h, y + h k3];
    deltay = h (1/6 k1 + 1/3 k2 + 1/3 k3 + 1/6 k4);
    {h, deltay}
  ];\
\>", "Input",
  CellLabel->"In[50]:="],

Cell["\<\
Notice that the implementation closely resembles the description \
that you might find in a textbook. There are no memory \
allocation/deallocation statements or type declarations, for example. In fact \
the implementation works for machine real numbers, machine complex numbers, \
and even using arbitrary precision software arithmetic.\
\>", "Text"],

Cell["\<\
Here is an example of the calling syntax. For simplicity the method \
only uses fixed step sizes so we need to specify what step sizes to \
take.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[system, Method \[Rule] ClassicalRungeKutta, 
      StartingStepSize \[Rule] 1/10]\)], "Input",
  CellLabel->"In[51]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 20.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[51]="]
}, Open  ]],

Cell[TextData[{
  "Many of the methods that have been built into ",
  StyleBox["NDSolve", "MR"],
  " were first prototyped using top-level code before being implemented in \
the kernel for efficiency."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness", "Section",
  CellTags->"s:6"],

Cell["\<\
Stiffness is a combination of problem, initial data, numerical \
method, and error tolerances.\
\>", "Text"],

Cell["\<\
Stiffness can arise, for example, in the translation of diffusion \
terms by divided differences into a large system of ODEs.\
\>", "Text"],

Cell["\<\
In order to understand more about the nature of stiffness it is \
useful to study how methods behave when applied to a simple problem.\
\>", \
"Text"],

Cell[CellGroupData[{

Cell["Linear stability", "Subsection"],

Cell["\<\
Consider applying a Runge-Kutta method to a linear scalar equation \
known as Dahlquist's equation:\
\>", "Text"],

Cell[BoxData[
    \(TraditionalForm\`y' \((t)\)\  = \ \[Lambda]\ \(y(
          t)\)\ , \ \[Lambda] \[Element] \[DoubleStruckCapitalC], \ 
    Re(\[Lambda])\  < \ 0. \)], "NumberedEquation",
  CellTags->"DahlquistEquation"],

Cell[TextData[{
  "The result is a rational function of polynomials ",
  Cell[BoxData[
      \(TraditionalForm\`R(z)\)]],
  " where ",
  Cell[BoxData[
      \(TraditionalForm\`z\  = \ h\ \[Lambda]\)]],
  " (see for example [",
  ButtonBox["L87",
    ButtonData:>{"References.nb", "L87"},
    ButtonStyle->"Hyperlink"],
  "])."
}], "Text"],

Cell[TextData[{
  "This utility function finds the linear stability function ",
  Cell[BoxData[
      \(TraditionalForm\`R(z)\)]],
  " for Runge-Kutta methods. The form depends on the coefficients and is a \
polynomial if the Runge-Kutta method is explicit."
}], "Text"],

Cell["\<\
Here is the stability function for the fifth-order scheme in the \
Dormand-Prince 5(4) pair.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DOPRIsf\  = \ 
      RungeKuttaLinearStabilityFunction[DOPRIamat, \ DOPRIbvec, z]\)], "Input",\

  CellLabel->"In[52]:="],

Cell[BoxData[
    \(1 + z + z\^2\/2 + z\^3\/6 + z\^4\/24 + z\^5\/120 + 
      z\^6\/600\)], "Output",
  CellLabel->"Out[52]="]
}, Open  ]],

Cell[TextData[{
  "This function finds the linear stability function ",
  Cell[BoxData[
      \(TraditionalForm\`R(z)\)]],
  " for Runge-Kutta methods. The form depends on the coefficients and is a \
polynomial if the Runge-Kutta method is explicit."
}], "Text"],

Cell["\<\
The following package is useful for visualizing linear stability \
regions for numerical methods for differential equations.\
\>", "MathCaption"],

Cell[BoxData[
    \(Needs["\<NumericalMath`OrderStar`\>"]; \)], "Input",
  CellLabel->"In[53]:="],

Cell[TextData[{
  "We can now visualize the absolute stability region ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(|\)\(R(z)\)\(|\)\)\  = \ 1\)]],
  "."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(OrderStar[DOPRIsf, \ 1, z]; \)], "Input",
  CellLabel->"In[54]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1.87597 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.599221 0.162446 0.937983 0.162446 [
[ 0 0 -0.125 0 ]
[ 0 0 -0.125 0 ]
[ 0 1.87597 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1.87597 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
0 0 m
1 0 L
s
0 0 m
0 1.87597 L
s
0 1.87597 m
1 1.87597 L
s
1 0 m
1 1.87597 L
s
0 .93798 m
1 .93798 L
s
.59922 0 m
.59922 1.87597 L
s
0 0 m
1 0 L
1 1.87597 L
0 1.87597 L
closepath
clip
newpath
1 g
.212 1.30605 m
.21519 1.30605 L
.21519 1.30976 L
F
.10649 1.16357 m
.11392 1.16357 L
.11392 1.17681 L
F
.72152 1.51469 m
.72152 1.51977 L
.71984 1.51977 L
F
.08576 1.11608 m
.08861 1.11608 L
.08861 1.12365 L
F
.10127 .72292 m
.10127 .73614 L
.09509 .73614 L
F
.77845 .32058 m
.77215 .32058 L
.77215 .31641 L
F
.27338 1.37729 m
.27848 1.37729 L
.27848 1.38179 L
F
.13595 1.21107 m
.13924 1.21107 L
.13924 1.2156 L
F
.56771 1.35355 m
.55696 1.36897 L
.55696 1.35355 L
F
.07284 1.06859 m
.07595 1.06859 L
.07595 1.08224 L
F
.23138 1.3298 m
.24051 1.3298 L
.24051 1.34018 L
F
.76546 1.49602 m
.75949 1.49602 L
.75949 1.49394 L
F
.12658 .67875 m
.12658 .68865 L
.12015 .68865 L
F
.72443 1.5079 m
.72785 1.5036 L
.72785 1.5079 L
F
.24051 .53579 m
.24051 .54617 L
.23138 .54617 L
F
.25129 1.35355 m
.25316 1.35355 L
.25316 1.35549 L
F
.13924 .66036 m
.13924 .6649 L
.13595 .6649 L
F
.73418 .32003 m
.73418 .32058 L
.73341 .32058 L
F
.30044 1.40104 m
.3038 1.40104 L
.3038 1.40333 L
F
.58639 1.30605 m
.58228 1.31943 L
.58228 1.30605 L
F
.52483 1.40104 m
.51899 1.40476 L
.51899 1.40104 L
F
.0627 .9736 m
.06329 .9736 L
.06329 .98683 L
F
.17285 1.25856 m
.17722 1.25856 L
.17722 1.26378 L
F
.17722 .61219 m
.17722 .61741 L
.17285 .61741 L
F
.21519 .5662 m
.21519 .56991 L
.212 .56991 L
F
.57892 1.3298 m
.56962 1.34967 L
.56962 1.3298 L
F
.59666 1.23481 m
.59494 1.2553 L
.59494 1.23481 L
F
.55078 1.37729 m
.5443 1.38361 L
.5443 1.37729 L
F
.78682 .33245 m
.78481 .33245 L
.78481 .32911 L
F
.56771 .52242 m
.55696 .52242 L
.55696 .507 L
F
.27848 .49417 m
.27848 .49867 L
.27338 .49867 L
F
.72785 .32703 m
.72785 .33245 L
.72372 .33245 L
F
.58639 .56991 m
.58228 .56991 L
.58228 .55654 L
F
.12015 1.18732 m
.12658 1.18732 L
.12658 1.19722 L
F
.52483 .47493 m
.51899 .47493 L
.51899 .4712 L
F
.48213 .45118 m
.48101 .45118 L
.48101 .45077 L
F
.57892 .54617 m
.56962 .54617 L
.56962 .52629 L
F
.77215 1.55956 m
.77215 1.55539 L
.77845 1.55539 L
F
.59666 .64115 m
.59494 .64115 L
.59494 .62067 L
F
.15366 1.23481 m
.16456 1.23481 L
.16456 1.24824 L
F
.20253 .58182 m
.20253 .59366 L
.19243 .59366 L
F
.72372 1.54352 m
.72785 1.54352 L
.72785 1.54893 L
F
.08861 .75232 m
.08861 .75989 L
.08576 .75989 L
F
.3038 .47264 m
.3038 .47493 L
.30044 .47493 L
F
.72152 .3393 m
.72152 .34432 L
.71976 .34432 L
F
.16456 .62773 m
.16456 .64115 L
.15366 .64115 L
F
.11392 .69915 m
.11392 .71239 L
.10649 .71239 L
F
.25316 .52048 m
.25316 .52242 L
.25129 .52242 L
F
.34041 1.42478 m
.34177 1.42478 L
.34177 1.42535 L
F
.09509 1.13983 m
.10127 1.13983 L
.10127 1.15305 L
F
.72785 .37236 m
.72443 .36807 L
.72785 .36807 L
F
.19243 1.28231 m
.20253 1.28231 L
.20253 1.29415 L
F
.06329 .88914 m
.06329 .90236 L
.0627 .90236 L
F
.78131 .36807 m
.77848 .37141 L
.77848 .36807 L
F
.78481 1.54686 m
.78481 1.54352 L
.78682 1.54352 L
F
.07595 .79373 m
.07595 .80738 L
.07284 .80738 L
F
.78771 1.51977 m
.78481 1.51977 L
.78481 1.51337 L
F
.75949 .38202 m
.75949 .37994 L
.76546 .37994 L
F
.71984 .3562 m
.72152 .3562 L
.72152 .36128 L
F
.48213 1.42478 m
.48101 1.4252 L
.48101 1.42478 L
F
.78481 .3626 m
.78481 .3562 L
.78771 .3562 L
F
.73341 1.55539 m
.73418 1.55539 L
.73418 1.55593 L
F
.78131 1.5079 m
.77848 1.5079 L
.77848 1.50456 L
F
.55078 .49867 m
.5443 .49867 L
.5443 .49236 L
F
.71976 1.53164 m
.72152 1.53164 L
.72152 1.53667 L
F
.34177 .45061 m
.34177 .45118 L
.34041 .45118 L
F
.59888 1.16357 m
.59854 1.18732 L
.59494 1.18732 L
.59494 1.16357 L
F
.46835 1.18732 m
.48101 1.18732 L
.48101 1.21107 L
.46835 1.21107 L
F
.44304 1.18732 m
.4557 1.18732 L
.4557 1.21107 L
.44304 1.21107 L
F
.22785 1.32521 m
.21519 1.30976 L
.21519 1.30605 L
.22785 1.30605 L
F
.53165 1.18732 m
.5443 1.18732 L
.5443 1.21107 L
.53165 1.21107 L
F
.53165 .78363 m
.5443 .78363 L
.5443 .80738 L
.53165 .80738 L
F
.5443 .78363 m
.55696 .78363 L
.55696 .80738 L
.5443 .80738 L
F
.55696 .78363 m
.56962 .78363 L
.56962 .80738 L
.55696 .80738 L
F
.46835 .78363 m
.48101 .78363 L
.48101 .80738 L
.46835 .80738 L
F
.48101 .78363 m
.49367 .78363 L
.49367 .80738 L
.48101 .80738 L
F
.49367 .78363 m
.50633 .78363 L
.50633 .80738 L
.49367 .80738 L
F
.20253 1.21107 m
.21519 1.21107 L
.21519 1.23481 L
.20253 1.23481 L
F
.21519 1.21107 m
.22785 1.21107 L
.22785 1.23481 L
.21519 1.23481 L
F
.24051 1.21107 m
.25316 1.21107 L
.25316 1.23481 L
.24051 1.23481 L
F
.50633 .78363 m
.51899 .78363 L
.51899 .80738 L
.50633 .80738 L
F
.51899 .78363 m
.53165 .78363 L
.53165 .80738 L
.51899 .80738 L
F
.51899 1.18732 m
.53165 1.18732 L
.53165 1.21107 L
.51899 1.21107 L
F
.59465 1.25856 m
.59135 1.28231 L
.58228 1.28231 L
.58228 1.25856 L
F
.77848 .37141 m
.77215 .3766 L
.77215 .36807 L
.77848 .36807 L
F
.50633 1.18732 m
.51899 1.18732 L
.51899 1.21107 L
.50633 1.21107 L
F
.59135 1.28231 m
.58639 1.30605 L
.58228 1.30605 L
.58228 1.28231 L
F
.18987 1.27916 m
.17722 1.26378 L
.17722 1.25856 L
.18987 1.25856 L
F
.77215 .3766 m
.76582 .37984 L
.76582 .36807 L
.77215 .36807 L
F
.73418 .37773 m
.72785 .37236 L
.72785 .36807 L
.73418 .36807 L
F
.74051 1.55968 m
.73418 1.55593 L
.73418 1.55539 L
.74051 1.55539 L
F
.56962 .78363 m
.58228 .78363 L
.58228 .80738 L
.56962 .80738 L
F
.77215 1.49937 m
.77848 1.50456 L
.77848 1.5079 L
.77215 1.5079 L
F
.77532 .33839 m
.77215 .33839 L
.77215 .33245 L
.77532 .33245 L
F
.75 .33839 m
.75316 .33839 L
.75316 .34432 L
.75 .34432 L
F
.76266 .33839 m
.76582 .33839 L
.76582 .34432 L
.76266 .34432 L
F
.77532 .33839 m
.77532 .33245 L
.77848 .33245 L
.77848 .33839 L
F
.76266 .33839 m
.76266 .33245 L
.76582 .33245 L
.76582 .33839 L
F
.76266 .33839 m
.75949 .33839 L
.75949 .33245 L
.76266 .33245 L
F
.75 .33839 m
.75 .33245 L
.75316 .33245 L
.75316 .33839 L
F
.75 .33839 m
.74684 .33839 L
.74684 .33245 L
.75 .33245 L
F
.73734 .33839 m
.74051 .33839 L
.74051 .34432 L
.73734 .34432 L
F
.73734 .33839 m
.73734 .33245 L
.74051 .33245 L
.74051 .33839 L
F
.39241 1.16357 m
.40506 1.16357 L
.40506 1.18732 L
.39241 1.18732 L
F
.73734 .33839 m
.73418 .33839 L
.73418 .33245 L
.73734 .33245 L
F
.40506 1.16357 m
.41772 1.16357 L
.41772 1.18732 L
.40506 1.18732 L
F
.34177 1.16357 m
.35443 1.16357 L
.35443 1.18732 L
.34177 1.18732 L
F
.36709 1.16357 m
.37975 1.16357 L
.37975 1.18732 L
.36709 1.18732 L
F
.75 1.55539 m
.74684 1.55539 L
.74684 1.54945 L
.75 1.54945 L
F
.76266 1.52571 m
.75949 1.52571 L
.75949 1.51977 L
.76266 1.51977 L
F
.76266 1.52571 m
.76582 1.52571 L
.76582 1.53164 L
.76266 1.53164 L
F
.35443 1.16357 m
.36709 1.16357 L
.36709 1.18732 L
.35443 1.18732 L
F
.32911 1.16357 m
.34177 1.16357 L
.34177 1.18732 L
.32911 1.18732 L
F
.29114 1.16357 m
.3038 1.16357 L
.3038 1.18732 L
.29114 1.18732 L
F
.31646 1.16357 m
.32911 1.16357 L
.32911 1.18732 L
.31646 1.18732 L
F
.37975 1.16357 m
.39241 1.16357 L
.39241 1.18732 L
.37975 1.18732 L
F
.76266 1.55539 m
.75949 1.55539 L
.75949 1.54945 L
.76266 1.54945 L
F
.3038 1.16357 m
.31646 1.16357 L
.31646 1.18732 L
.3038 1.18732 L
F
.50633 1.41233 m
.49367 1.41919 L
.49367 1.40104 L
.50633 1.40104 L
F
.31646 1.4114 m
.3038 1.40333 L
.3038 1.40104 L
.31646 1.40104 L
F
.48101 1.18732 m
.49367 1.18732 L
.49367 1.21107 L
.48101 1.21107 L
F
.51899 1.40476 m
.50633 1.41233 L
.50633 1.40104 L
.51899 1.40104 L
F
.32911 1.41887 m
.31646 1.4114 L
.31646 1.40104 L
.32911 1.40104 L
F
.39241 1.43791 m
.37975 1.43615 L
.37975 1.42478 L
.39241 1.42478 L
F
.58228 1.18732 m
.59494 1.18732 L
.59494 1.21107 L
.58228 1.21107 L
F
.59904 1.13983 m
.59888 1.16357 L
.59494 1.16357 L
.59494 1.13983 L
F
.59786 1.21107 m
.59666 1.23481 L
.59494 1.23481 L
.59494 1.21107 L
F
.4557 1.18732 m
.46835 1.18732 L
.46835 1.21107 L
.4557 1.21107 L
F
.59854 1.18732 m
.59786 1.21107 L
.59494 1.21107 L
.59494 1.18732 L
F
.1519 1.23254 m
.13924 1.2156 L
.13924 1.21107 L
.1519 1.21107 L
F
.49367 1.18732 m
.50633 1.18732 L
.50633 1.21107 L
.49367 1.21107 L
F
.26582 1.36887 m
.25316 1.35549 L
.25316 1.35355 L
.26582 1.35355 L
F
.5443 1.18732 m
.55696 1.18732 L
.55696 1.21107 L
.5443 1.21107 L
F
.55696 1.18732 m
.56962 1.18732 L
.56962 1.21107 L
.55696 1.21107 L
F
.29114 1.39292 m
.27848 1.38179 L
.27848 1.37729 L
.29114 1.37729 L
F
.56962 1.18732 m
.58228 1.18732 L
.58228 1.21107 L
.56962 1.21107 L
F
.5443 1.38361 m
.53165 1.39518 L
.53165 1.37729 L
.5443 1.37729 L
F
.41772 1.4386 m
.40506 1.43872 L
.40506 1.42478 L
.41772 1.42478 L
F
.40506 1.43872 m
.39241 1.43791 L
.39241 1.42478 L
.40506 1.42478 L
F
.37975 1.43615 m
.36709 1.43344 L
.36709 1.42478 L
.37975 1.42478 L
F
.36709 1.43344 m
.35443 1.42982 L
.35443 1.42478 L
.36709 1.42478 L
F
.35443 1.42982 m
.34177 1.42535 L
.34177 1.42478 L
.35443 1.42478 L
F
.26582 1.21107 m
.27848 1.21107 L
.27848 1.23481 L
.26582 1.23481 L
F
.32911 1.18732 m
.34177 1.18732 L
.34177 1.21107 L
.32911 1.21107 L
F
.29114 .71239 m
.3038 .71239 L
.3038 .73614 L
.29114 .73614 L
F
.31646 1.18732 m
.32911 1.18732 L
.32911 1.21107 L
.31646 1.21107 L
F
.59911 .92611 m
.59911 .94986 L
.59494 .94986 L
.59494 .92611 L
F
.3038 .71239 m
.31646 .71239 L
.31646 .73614 L
.3038 .73614 L
F
.55696 1.16357 m
.56962 1.16357 L
.56962 1.18732 L
.55696 1.18732 L
F
.31646 .71239 m
.32911 .71239 L
.32911 .73614 L
.31646 .73614 L
F
.27848 1.18732 m
.29114 1.18732 L
.29114 1.21107 L
.27848 1.21107 L
F
.59911 .94986 m
.59911 .9736 L
.59494 .9736 L
.59494 .94986 L
F
.26582 1.18732 m
.27848 1.18732 L
.27848 1.21107 L
.26582 1.21107 L
F
.59911 .85487 m
.59911 .87862 L
.59494 .87862 L
.59494 .85487 L
F
.35443 1.18732 m
.36709 1.18732 L
.36709 1.21107 L
.35443 1.21107 L
F
.59911 .90236 m
.59911 .92611 L
.59494 .92611 L
.59494 .90236 L
F
.34177 1.18732 m
.35443 1.18732 L
.35443 1.21107 L
.34177 1.21107 L
F
.35443 .71239 m
.36709 .71239 L
.36709 .73614 L
.35443 .73614 L
F
.44304 1.16357 m
.4557 1.16357 L
.4557 1.18732 L
.44304 1.18732 L
F
.32911 .71239 m
.34177 .71239 L
.34177 .73614 L
.32911 .73614 L
F
.36709 .71239 m
.37975 .71239 L
.37975 .73614 L
.36709 .73614 L
F
.4557 1.16357 m
.46835 1.16357 L
.46835 1.18732 L
.4557 1.18732 L
F
.76899 1.54945 m
.76582 1.54945 L
.76582 1.54352 L
.76899 1.54352 L
F
.46835 1.16357 m
.48101 1.16357 L
.48101 1.18732 L
.46835 1.18732 L
F
.56962 1.16357 m
.58228 1.16357 L
.58228 1.18732 L
.56962 1.18732 L
F
.37975 .71239 m
.39241 .71239 L
.39241 .73614 L
.37975 .73614 L
F
.5443 1.16357 m
.55696 1.16357 L
.55696 1.18732 L
.5443 1.18732 L
F
.58228 1.16357 m
.59494 1.16357 L
.59494 1.18732 L
.58228 1.18732 L
F
.39241 .71239 m
.40506 .71239 L
.40506 .73614 L
.39241 .73614 L
F
.34177 .71239 m
.35443 .71239 L
.35443 .73614 L
.34177 .73614 L
F
.40506 .71239 m
.41772 .71239 L
.41772 .73614 L
.40506 .73614 L
F
.59666 .64115 m
.59786 .6649 L
.59494 .6649 L
.59494 .64115 L
F
.59911 1.0211 m
.59911 1.04484 L
.59494 1.04484 L
.59494 1.0211 L
F
.29114 1.21107 m
.3038 1.21107 L
.3038 1.23481 L
.29114 1.23481 L
F
.3038 1.21107 m
.31646 1.21107 L
.31646 1.23481 L
.3038 1.23481 L
F
.39241 1.18732 m
.40506 1.18732 L
.40506 1.21107 L
.39241 1.21107 L
F
.36709 1.18732 m
.37975 1.18732 L
.37975 1.21107 L
.36709 1.21107 L
F
.59911 1.04484 m
.59911 1.06859 L
.59494 1.06859 L
.59494 1.04484 L
F
.35443 1.21107 m
.36709 1.21107 L
.36709 1.23481 L
.35443 1.23481 L
F
.59786 .6649 m
.59854 .68865 L
.59494 .68865 L
.59494 .6649 L
F
.36709 1.21107 m
.37975 1.21107 L
.37975 1.23481 L
.36709 1.23481 L
F
.34177 1.21107 m
.35443 1.21107 L
.35443 1.23481 L
.34177 1.23481 L
F
.1519 1.18732 m
.16456 1.18732 L
.16456 1.21107 L
.1519 1.21107 L
F
.13924 1.18732 m
.1519 1.18732 L
.1519 1.21107 L
.13924 1.21107 L
F
.27848 1.21107 m
.29114 1.21107 L
.29114 1.23481 L
.27848 1.23481 L
F
.59911 1.06859 m
.59911 1.09233 L
.59494 1.09233 L
.59494 1.06859 L
F
.29114 1.18732 m
.3038 1.18732 L
.3038 1.21107 L
.29114 1.21107 L
F
.40506 1.18732 m
.41772 1.18732 L
.41772 1.21107 L
.40506 1.21107 L
F
.41772 1.18732 m
.43038 1.18732 L
.43038 1.21107 L
.41772 1.21107 L
F
.3038 1.18732 m
.31646 1.18732 L
.31646 1.21107 L
.3038 1.21107 L
F
.59911 .83112 m
.59911 .85487 L
.59494 .85487 L
.59494 .83112 L
F
.59911 .87862 m
.59911 .90236 L
.59494 .90236 L
.59494 .87862 L
F
.37975 1.18732 m
.39241 1.18732 L
.39241 1.21107 L
.37975 1.21107 L
F
.59911 .99735 m
.59911 1.0211 L
.59494 1.0211 L
.59494 .99735 L
F
.59911 .9736 m
.59911 .99735 L
.59494 .99735 L
.59494 .9736 L
F
.59911 1.09233 m
.5991 1.11608 L
.59494 1.11608 L
.59494 1.09233 L
F
.5991 1.11608 m
.59904 1.13983 L
.59494 1.13983 L
.59494 1.11608 L
F
.43038 1.18732 m
.44304 1.18732 L
.44304 1.21107 L
.43038 1.21107 L
F
.73418 .32003 m
.74051 .31629 L
.74051 .32058 L
.73418 .32058 L
F
.51899 .71239 m
.53165 .71239 L
.53165 .73614 L
.51899 .73614 L
F
.50633 .71239 m
.51899 .71239 L
.51899 .73614 L
.50633 .73614 L
F
.78481 1.54686 m
.77848 1.55536 L
.77848 1.54352 L
.78481 1.54352 L
F
.58228 .71239 m
.59494 .71239 L
.59494 .73614 L
.58228 .73614 L
F
.5443 .71239 m
.55696 .71239 L
.55696 .73614 L
.5443 .73614 L
F
.58228 .78363 m
.59494 .78363 L
.59494 .80738 L
.58228 .80738 L
F
.55696 1.21107 m
.56962 1.21107 L
.56962 1.23481 L
.55696 1.23481 L
F
.53165 1.21107 m
.5443 1.21107 L
.5443 1.23481 L
.53165 1.23481 L
F
.43038 .71239 m
.44304 .71239 L
.44304 .73614 L
.43038 .73614 L
F
.5443 1.21107 m
.55696 1.21107 L
.55696 1.23481 L
.5443 1.23481 L
F
.56962 1.21107 m
.58228 1.21107 L
.58228 1.23481 L
.56962 1.23481 L
F
.58228 1.21107 m
.59494 1.21107 L
.59494 1.23481 L
.58228 1.23481 L
F
.49367 .71239 m
.50633 .71239 L
.50633 .73614 L
.49367 .73614 L
F
.4557 1.43291 m
.44304 1.43565 L
.44304 1.42478 L
.4557 1.42478 L
F
.17722 1.21107 m
.18987 1.21107 L
.18987 1.23481 L
.17722 1.23481 L
F
.43038 1.43756 m
.41772 1.4386 L
.41772 1.42478 L
.43038 1.42478 L
F
.46835 1.42941 m
.4557 1.43291 L
.4557 1.42478 L
.46835 1.42478 L
F
.16456 1.21107 m
.17722 1.21107 L
.17722 1.23481 L
.16456 1.23481 L
F
.22785 1.21107 m
.24051 1.21107 L
.24051 1.23481 L
.22785 1.23481 L
F
.18987 1.21107 m
.20253 1.21107 L
.20253 1.23481 L
.18987 1.23481 L
F
.56962 .71239 m
.58228 .71239 L
.58228 .73614 L
.56962 .73614 L
F
.55696 .71239 m
.56962 .71239 L
.56962 .73614 L
.55696 .73614 L
F
.53165 .71239 m
.5443 .71239 L
.5443 .73614 L
.53165 .73614 L
F
.25316 1.21107 m
.26582 1.21107 L
.26582 1.23481 L
.25316 1.23481 L
F
.44304 1.43565 m
.43038 1.43756 L
.43038 1.42478 L
.44304 1.42478 L
F
.48101 1.4252 m
.46835 1.42941 L
.46835 1.42478 L
.48101 1.42478 L
F
.48101 1.21107 m
.49367 1.21107 L
.49367 1.23481 L
.48101 1.23481 L
F
.4557 1.21107 m
.46835 1.21107 L
.46835 1.23481 L
.4557 1.23481 L
F
.74684 .38254 m
.73855 .37994 L
.74051 .37994 L
.74684 .37994 L
F
.49367 1.16357 m
.50633 1.16357 L
.50633 1.18732 L
.49367 1.18732 L
F
.50633 1.21107 m
.51899 1.21107 L
.51899 1.23481 L
.50633 1.23481 L
F
.51899 1.21107 m
.53165 1.21107 L
.53165 1.23481 L
.51899 1.23481 L
F
.46835 1.21107 m
.48101 1.21107 L
.48101 1.23481 L
.46835 1.23481 L
F
.41772 1.16357 m
.43038 1.16357 L
.43038 1.18732 L
.41772 1.18732 L
F
.50633 1.16357 m
.51899 1.16357 L
.51899 1.18732 L
.50633 1.18732 L
F
.76899 1.54945 m
.76899 1.55539 L
.76582 1.55539 L
.76582 1.54945 L
F
.43038 1.16357 m
.44304 1.16357 L
.44304 1.18732 L
.43038 1.18732 L
F
.51899 1.16357 m
.53165 1.16357 L
.53165 1.18732 L
.51899 1.18732 L
F
.48101 1.16357 m
.49367 1.16357 L
.49367 1.18732 L
.48101 1.18732 L
F
.53165 1.16357 m
.5443 1.16357 L
.5443 1.18732 L
.53165 1.18732 L
F
.4557 .71239 m
.46835 .71239 L
.46835 .73614 L
.4557 .73614 L
F
.48101 .71239 m
.49367 .71239 L
.49367 .73614 L
.48101 .73614 L
F
.41772 1.21107 m
.43038 1.21107 L
.43038 1.23481 L
.41772 1.23481 L
F
.43038 1.21107 m
.44304 1.21107 L
.44304 1.23481 L
.43038 1.23481 L
F
.41772 .71239 m
.43038 .71239 L
.43038 .73614 L
.41772 .73614 L
F
.44304 .71239 m
.4557 .71239 L
.4557 .73614 L
.44304 .73614 L
F
.46835 .71239 m
.48101 .71239 L
.48101 .73614 L
.46835 .73614 L
F
.58639 .56991 m
.59135 .59366 L
.58228 .59366 L
.58228 .56991 L
F
.40506 1.21107 m
.41772 1.21107 L
.41772 1.23481 L
.40506 1.23481 L
F
.49367 1.21107 m
.50633 1.21107 L
.50633 1.23481 L
.49367 1.23481 L
F
.39241 1.21107 m
.40506 1.21107 L
.40506 1.23481 L
.39241 1.23481 L
F
.44304 1.21107 m
.4557 1.21107 L
.4557 1.23481 L
.44304 1.23481 L
F
.37975 1.21107 m
.39241 1.21107 L
.39241 1.23481 L
.37975 1.23481 L
F
.74367 .32651 m
.74684 .32651 L
.74684 .33245 L
.74367 .33245 L
F
.76266 1.54945 m
.76582 1.54945 L
.76582 1.55539 L
.76266 1.55539 L
F
.75633 .32651 m
.75316 .32651 L
.75316 .32058 L
.75633 .32058 L
F
.75633 .32651 m
.75633 .33245 L
.75316 .33245 L
.75316 .32651 L
F
.74367 .32651 m
.74367 .33245 L
.74051 .33245 L
.74051 .32651 L
F
.76899 .32651 m
.76582 .32651 L
.76582 .32058 L
.76899 .32058 L
F
.74367 .32651 m
.74051 .32651 L
.74051 .32058 L
.74367 .32058 L
F
.73734 .35026 m
.73418 .35026 L
.73418 .34432 L
.73734 .34432 L
F
.73734 .35026 m
.73734 .3562 L
.73418 .3562 L
.73418 .35026 L
F
.75633 .35026 m
.75633 .3562 L
.75316 .3562 L
.75316 .35026 L
F
.75633 .35026 m
.75949 .35026 L
.75949 .3562 L
.75633 .3562 L
F
.73734 .35026 m
.73734 .34432 L
.74051 .34432 L
.74051 .35026 L
F
.75633 .32651 m
.75949 .32651 L
.75949 .33245 L
.75633 .33245 L
F
.75633 .33839 m
.75949 .33839 L
.75949 .34432 L
.75633 .34432 L
F
.38291 .54023 m
.37975 .54023 L
.37975 .53429 L
.38291 .53429 L
F
.38924 .54023 m
.39241 .54023 L
.39241 .54617 L
.38924 .54617 L
F
.38291 .54023 m
.38291 .54617 L
.37975 .54617 L
.37975 .54023 L
F
.38291 .54023 m
.38291 .53429 L
.38608 .53429 L
.38608 .54023 L
F
.38924 .54023 m
.38608 .54023 L
.38608 .53429 L
.38924 .53429 L
F
.38924 .54023 m
.38924 .54617 L
.38608 .54617 L
.38608 .54023 L
F
.77532 .35026 m
.77532 .34432 L
.77848 .34432 L
.77848 .35026 L
F
.76266 .32651 m
.76266 .33245 L
.75949 .33245 L
.75949 .32651 L
F
.75 .32651 m
.74684 .32651 L
.74684 .32058 L
.75 .32058 L
F
.76899 .32651 m
.76899 .33245 L
.76582 .33245 L
.76582 .32651 L
F
.76266 .32651 m
.75949 .32651 L
.75949 .32058 L
.76266 .32058 L
F
.76899 .32651 m
.77215 .32651 L
.77215 .33245 L
.76899 .33245 L
F
.75 .32651 m
.75 .32058 L
.75316 .32058 L
.75316 .32651 L
F
.75 .32651 m
.75 .33245 L
.74684 .33245 L
.74684 .32651 L
F
.75633 1.52571 m
.75949 1.52571 L
.75949 1.53164 L
.75633 1.53164 L
F
.76899 1.52571 m
.76899 1.53164 L
.76582 1.53164 L
.76582 1.52571 L
F
.73734 1.53758 m
.73418 1.53758 L
.73418 1.53164 L
.73734 1.53164 L
F
.75633 1.52571 m
.75633 1.53164 L
.75316 1.53164 L
.75316 1.52571 L
F
.76899 1.53758 m
.77215 1.53758 L
.77215 1.54352 L
.76899 1.54352 L
F
.76899 1.52571 m
.76582 1.52571 L
.76582 1.51977 L
.76899 1.51977 L
F
.76899 1.52571 m
.77215 1.52571 L
.77215 1.53164 L
.76899 1.53164 L
F
.74367 1.54945 m
.74367 1.55539 L
.74051 1.55539 L
.74051 1.54945 L
F
.75633 1.52571 m
.75316 1.52571 L
.75316 1.51977 L
.75633 1.51977 L
F
.77532 1.53758 m
.77532 1.53164 L
.77848 1.53164 L
.77848 1.53758 L
F
.74367 1.54945 m
.74051 1.54945 L
.74051 1.54352 L
.74367 1.54352 L
F
.75633 1.53758 m
.75949 1.53758 L
.75949 1.54352 L
.75633 1.54352 L
F
.73734 1.53758 m
.73734 1.53164 L
.74051 1.53164 L
.74051 1.53758 L
F
.75633 1.53758 m
.75633 1.54352 L
.75316 1.54352 L
.75316 1.53758 L
F
.76266 .32651 m
.76266 .32058 L
.76582 .32058 L
.76582 .32651 L
F
.74367 .33839 m
.74051 .33839 L
.74051 .33245 L
.74367 .33245 L
F
.74367 .35026 m
.74367 .3562 L
.74051 .3562 L
.74051 .35026 L
F
.74367 .33839 m
.74684 .33839 L
.74684 .34432 L
.74367 .34432 L
F
.75633 .33839 m
.75316 .33839 L
.75316 .33245 L
.75633 .33245 L
F
.75633 .33839 m
.75633 .34432 L
.75316 .34432 L
.75316 .33839 L
F
.74367 .33839 m
.74367 .34432 L
.74051 .34432 L
.74051 .33839 L
F
.75633 1.53758 m
.75316 1.53758 L
.75316 1.53164 L
.75633 1.53164 L
F
.75 1.53758 m
.75 1.53164 L
.75316 1.53164 L
.75316 1.53758 L
F
.75 1.53758 m
.75 1.54352 L
.74684 1.54352 L
.74684 1.53758 L
F
.74367 .35026 m
.74684 .35026 L
.74684 .3562 L
.74367 .3562 L
F
.74367 .35026 m
.74051 .35026 L
.74051 .34432 L
.74367 .34432 L
F
.75 1.53758 m
.74684 1.53758 L
.74684 1.53164 L
.75 1.53164 L
F
.4557 .61741 m
.46835 .61741 L
.46835 .64115 L
.4557 .64115 L
F
.55696 .85487 m
.56962 .85487 L
.56962 .87862 L
.55696 .87862 L
F
.43038 .61741 m
.44304 .61741 L
.44304 .64115 L
.43038 .64115 L
F
.46835 .61741 m
.48101 .61741 L
.48101 .64115 L
.46835 .64115 L
F
.56962 .85487 m
.58228 .85487 L
.58228 .87862 L
.56962 .87862 L
F
.51899 .85487 m
.53165 .85487 L
.53165 .87862 L
.51899 .87862 L
F
.58228 .85487 m
.59494 .85487 L
.59494 .87862 L
.58228 .87862 L
F
.36709 .61741 m
.37975 .61741 L
.37975 .64115 L
.36709 .64115 L
F
.48101 .61741 m
.49367 .61741 L
.49367 .64115 L
.48101 .64115 L
F
.34177 .61741 m
.35443 .61741 L
.35443 .64115 L
.34177 .64115 L
F
.37975 .61741 m
.39241 .61741 L
.39241 .64115 L
.37975 .64115 L
F
.49367 .61741 m
.50633 .61741 L
.50633 .64115 L
.49367 .64115 L
F
.44304 .61741 m
.4557 .61741 L
.4557 .64115 L
.44304 .64115 L
F
.50633 .61741 m
.51899 .61741 L
.51899 .64115 L
.50633 .64115 L
F
.27848 .61741 m
.29114 .61741 L
.29114 .64115 L
.27848 .64115 L
F
.26582 .61741 m
.27848 .61741 L
.27848 .64115 L
.26582 .64115 L
F
.25316 .61741 m
.26582 .61741 L
.26582 .64115 L
.25316 .64115 L
F
.24051 .61741 m
.25316 .61741 L
.25316 .64115 L
.24051 .64115 L
F
.13924 .87862 m
.1519 .87862 L
.1519 .90236 L
.13924 .90236 L
F
.12658 .87862 m
.13924 .87862 L
.13924 .90236 L
.12658 .90236 L
F
.11392 .87862 m
.12658 .87862 L
.12658 .90236 L
.11392 .90236 L
F
.53165 .85487 m
.5443 .85487 L
.5443 .87862 L
.53165 .87862 L
F
.3038 .61741 m
.31646 .61741 L
.31646 .64115 L
.3038 .64115 L
F
.50633 .85487 m
.51899 .85487 L
.51899 .87862 L
.50633 .87862 L
F
.5443 .85487 m
.55696 .85487 L
.55696 .87862 L
.5443 .87862 L
F
.32911 .61741 m
.34177 .61741 L
.34177 .64115 L
.32911 .64115 L
F
.31646 .61741 m
.32911 .61741 L
.32911 .64115 L
.31646 .64115 L
F
.29114 .61741 m
.3038 .61741 L
.3038 .64115 L
.29114 .64115 L
F
.76899 .33839 m
.76582 .33839 L
.76582 .33245 L
.76899 .33245 L
F
.76899 .33839 m
.76899 .34432 L
.76582 .34432 L
.76582 .33839 L
F
.4019 .54023 m
.39873 .54023 L
.39873 .53429 L
.4019 .53429 L
F
.76899 .33839 m
.77215 .33839 L
.77215 .34432 L
.76899 .34432 L
F
.75633 .35026 m
.75316 .35026 L
.75316 .34432 L
.75633 .34432 L
F
.75 .35026 m
.75 .34432 L
.75316 .34432 L
.75316 .35026 L
F
.75 .35026 m
.75 .3562 L
.74684 .3562 L
.74684 .35026 L
F
.38924 .5521 m
.38924 .55804 L
.38608 .55804 L
.38608 .5521 L
F
.38924 .5521 m
.39241 .5521 L
.39241 .55804 L
.38924 .55804 L
F
.39557 .54023 m
.39557 .53429 L
.39873 .53429 L
.39873 .54023 L
F
.39557 .54023 m
.39241 .54023 L
.39241 .53429 L
.39557 .53429 L
F
.4019 .54023 m
.40506 .54023 L
.40506 .54617 L
.4019 .54617 L
F
.39557 .54023 m
.39557 .54617 L
.39241 .54617 L
.39241 .54023 L
F
.4019 .54023 m
.4019 .54617 L
.39873 .54617 L
.39873 .54023 L
F
.76899 .35026 m
.76582 .35026 L
.76582 .34432 L
.76899 .34432 L
F
.39241 .61741 m
.40506 .61741 L
.40506 .64115 L
.39241 .64115 L
F
.76899 .35026 m
.77215 .35026 L
.77215 .3562 L
.76899 .3562 L
F
.76266 .35026 m
.76266 .3562 L
.75949 .3562 L
.75949 .35026 L
F
.40506 .61741 m
.41772 .61741 L
.41772 .64115 L
.40506 .64115 L
F
.35443 .61741 m
.36709 .61741 L
.36709 .64115 L
.35443 .64115 L
F
.41772 .61741 m
.43038 .61741 L
.43038 .64115 L
.41772 .64115 L
F
.76266 .35026 m
.76266 .34432 L
.76582 .34432 L
.76582 .35026 L
F
.76266 .35026 m
.75949 .35026 L
.75949 .34432 L
.76266 .34432 L
F
.75 .35026 m
.74684 .35026 L
.74684 .34432 L
.75 .34432 L
F
.77532 .35026 m
.77532 .3562 L
.77215 .3562 L
.77215 .35026 L
F
.76899 .35026 m
.76899 .3562 L
.76582 .3562 L
.76582 .35026 L
F
.77532 .35026 m
.77215 .35026 L
.77215 .34432 L
.77532 .34432 L
F
.37342 1.34167 m
.37342 1.35355 L
.36709 1.35355 L
.36709 1.34167 L
F
.37342 1.34167 m
.37975 1.34167 L
.37975 1.35355 L
.37342 1.35355 L
F
.1519 1.16357 m
.16456 1.16357 L
.16456 1.18732 L
.1519 1.18732 L
F
.41139 1.31793 m
.40506 1.31793 L
.40506 1.30605 L
.41139 1.30605 L
F
.41139 1.31793 m
.41772 1.31793 L
.41772 1.3298 L
.41139 1.3298 L
F
.41139 1.34167 m
.41772 1.34167 L
.41772 1.35355 L
.41139 1.35355 L
F
.39873 1.32386 m
.39873 1.3298 L
.39557 1.3298 L
.39557 1.32386 L
F
.12658 1.16357 m
.13924 1.16357 L
.13924 1.18732 L
.12658 1.18732 L
F
.72785 1.5079 m
.73418 1.5079 L
.73418 1.51977 L
.72785 1.51977 L
F
.38291 1.33574 m
.38608 1.33574 L
.38608 1.34167 L
.38291 1.34167 L
F
.27848 .71239 m
.29114 .71239 L
.29114 .73614 L
.27848 .73614 L
F
.75316 1.5079 m
.74684 1.5079 L
.74684 1.49602 L
.75316 1.49602 L
F
.13924 1.16357 m
.1519 1.16357 L
.1519 1.18732 L
.13924 1.18732 L
F
.41139 1.34167 m
.41139 1.35355 L
.40506 1.35355 L
.40506 1.34167 L
F
.25316 1.16357 m
.26582 1.16357 L
.26582 1.18732 L
.25316 1.18732 L
F
.7894 1.53164 m
.78481 1.53164 L
.78481 1.51977 L
.78771 1.51977 L
F
.76899 1.53164 m
.77215 1.53164 L
.77215 1.53758 L
.76899 1.53758 L
F
.75 1.53164 m
.74684 1.53164 L
.74684 1.52571 L
.75 1.52571 L
F
.26582 1.16357 m
.27848 1.16357 L
.27848 1.18732 L
.26582 1.18732 L
F
.77532 1.53164 m
.77215 1.53164 L
.77215 1.52571 L
.77532 1.52571 L
F
.27848 1.16357 m
.29114 1.16357 L
.29114 1.18732 L
.27848 1.18732 L
F
.38291 1.34167 m
.37975 1.34167 L
.37975 1.33574 L
.38291 1.33574 L
F
.76582 1.53758 m
.76582 1.54352 L
.76266 1.54352 L
.76266 1.53758 L
F
.39557 1.34167 m
.39241 1.34167 L
.39241 1.33574 L
.39557 1.33574 L
F
.37342 1.34167 m
.36709 1.34167 L
.36709 1.3298 L
.37342 1.3298 L
F
.75316 1.53758 m
.75316 1.54352 L
.75 1.54352 L
.75 1.53758 L
F
.75633 1.53164 m
.75949 1.53164 L
.75949 1.53758 L
.75633 1.53758 L
F
.76266 1.53164 m
.75949 1.53164 L
.75949 1.52571 L
.76266 1.52571 L
F
.77532 1.52571 m
.77532 1.51977 L
.77848 1.51977 L
.77848 1.52571 L
F
.77532 1.52571 m
.77215 1.52571 L
.77215 1.51977 L
.77532 1.51977 L
F
.76266 1.54945 m
.76266 1.54352 L
.76582 1.54352 L
.76582 1.54945 L
F
.76266 1.54945 m
.75949 1.54945 L
.75949 1.54352 L
.76266 1.54352 L
F
.77532 1.52571 m
.77848 1.52571 L
.77848 1.53164 L
.77532 1.53164 L
F
.39557 .5521 m
.39241 .5521 L
.39241 .54617 L
.39557 .54617 L
F
.75 1.54945 m
.74684 1.54945 L
.74684 1.54352 L
.75 1.54352 L
F
.73734 1.52571 m
.74051 1.52571 L
.74051 1.53164 L
.73734 1.53164 L
F
.75 1.52571 m
.75316 1.52571 L
.75316 1.53164 L
.75 1.53164 L
F
.76266 1.52571 m
.76266 1.51977 L
.76582 1.51977 L
.76582 1.52571 L
F
.75 1.52571 m
.74684 1.52571 L
.74684 1.51977 L
.75 1.51977 L
F
.75 1.54945 m
.75316 1.54945 L
.75316 1.55539 L
.75 1.55539 L
F
.75 1.54945 m
.75 1.54352 L
.75316 1.54352 L
.75316 1.54945 L
F
.75 1.52571 m
.75 1.51977 L
.75316 1.51977 L
.75316 1.52571 L
F
.73734 1.52571 m
.73734 1.51977 L
.74051 1.51977 L
.74051 1.52571 L
F
.73734 1.52571 m
.73418 1.52571 L
.73418 1.51977 L
.73734 1.51977 L
F
.39557 1.33574 m
.39873 1.33574 L
.39873 1.34167 L
.39557 1.34167 L
F
.39557 1.33574 m
.39557 1.3298 L
.39873 1.3298 L
.39873 1.33574 L
F
.38291 1.33574 m
.37975 1.33574 L
.37975 1.3298 L
.38291 1.3298 L
F
.39557 1.33574 m
.39241 1.33574 L
.39241 1.3298 L
.39557 1.3298 L
F
.38291 1.33574 m
.38291 1.3298 L
.38608 1.3298 L
.38608 1.33574 L
F
.39557 .5521 m
.39557 .54617 L
.39873 .54617 L
.39873 .5521 L
F
.38291 .5521 m
.37975 .5521 L
.37975 .54617 L
.38291 .54617 L
F
.39557 .5521 m
.39873 .5521 L
.39873 .55804 L
.39557 .55804 L
F
.38291 .5521 m
.38608 .5521 L
.38608 .55804 L
.38291 .55804 L
F
.38291 .5521 m
.38291 .54617 L
.38608 .54617 L
.38608 .5521 L
F
.77532 .33839 m
.77848 .33839 L
.77848 .34432 L
.77532 .34432 L
F
.39557 1.32386 m
.39241 1.32386 L
.39241 1.31793 L
.39557 1.31793 L
F
.39557 1.32386 m
.39557 1.3298 L
.39241 1.3298 L
.39241 1.32386 L
F
.4019 1.32386 m
.39873 1.32386 L
.39873 1.31793 L
.4019 1.31793 L
F
.4019 1.32386 m
.4019 1.3298 L
.39873 1.3298 L
.39873 1.32386 L
F
.38291 1.32386 m
.38291 1.3298 L
.37975 1.3298 L
.37975 1.32386 L
F
.4019 1.32386 m
.40506 1.32386 L
.40506 1.3298 L
.4019 1.3298 L
F
.38291 1.32386 m
.37975 1.32386 L
.37975 1.31793 L
.38291 1.31793 L
F
.38924 1.32386 m
.38924 1.3298 L
.38608 1.3298 L
.38608 1.32386 L
F
.4019 .5521 m
.40506 .5521 L
.40506 .55804 L
.4019 .55804 L
F
.38924 1.32386 m
.39241 1.32386 L
.39241 1.3298 L
.38924 1.3298 L
F
.38924 1.32386 m
.38608 1.32386 L
.38608 1.31793 L
.38924 1.31793 L
F
.4019 .5521 m
.39873 .5521 L
.39873 .54617 L
.4019 .54617 L
F
.4019 .5521 m
.4019 .55804 L
.39873 .55804 L
.39873 .5521 L
F
.38924 .5521 m
.38608 .5521 L
.38608 .54617 L
.38924 .54617 L
F
.75633 1.54945 m
.75633 1.55539 L
.75316 1.55539 L
.75316 1.54945 L
F
.75633 1.54945 m
.75949 1.54945 L
.75949 1.55539 L
.75633 1.55539 L
F
.76266 1.53758 m
.75949 1.53758 L
.75949 1.53164 L
.76266 1.53164 L
F
.76266 1.53758 m
.76266 1.54352 L
.75949 1.54352 L
.75949 1.53758 L
F
.75633 1.54945 m
.75316 1.54945 L
.75316 1.54352 L
.75633 1.54352 L
F
.74367 1.54945 m
.74684 1.54945 L
.74684 1.55539 L
.74367 1.55539 L
F
.76899 1.54945 m
.77215 1.54945 L
.77215 1.55539 L
.76899 1.55539 L
F
.76266 1.53758 m
.76266 1.53164 L
.76582 1.53164 L
.76582 1.53758 L
F
.77532 1.53758 m
.77215 1.53758 L
.77215 1.53164 L
.77532 1.53164 L
F
.38291 1.32386 m
.38291 1.31793 L
.38608 1.31793 L
.38608 1.32386 L
F
.76899 1.53758 m
.76582 1.53758 L
.76582 1.53164 L
.76899 1.53164 L
F
.76899 1.53758 m
.76899 1.54352 L
.76582 1.54352 L
.76582 1.53758 L
F
.77532 1.53758 m
.77532 1.54352 L
.77215 1.54352 L
.77215 1.53758 L
F
.74051 1.53758 m
.74051 1.54352 L
.73734 1.54352 L
.73734 1.53758 L
F
.21519 1.16357 m
.22785 1.16357 L
.22785 1.18732 L
.21519 1.18732 L
F
.72785 1.53164 m
.72152 1.53164 L
.72152 1.51977 L
.72785 1.51977 L
F
.73734 1.53164 m
.73418 1.53164 L
.73418 1.52571 L
.73734 1.52571 L
F
.22785 1.16357 m
.24051 1.16357 L
.24051 1.18732 L
.22785 1.18732 L
F
.4019 1.33574 m
.40506 1.33574 L
.40506 1.34167 L
.4019 1.34167 L
F
.20253 1.16357 m
.21519 1.16357 L
.21519 1.18732 L
.20253 1.18732 L
F
.24051 1.16357 m
.25316 1.16357 L
.25316 1.18732 L
.24051 1.18732 L
F
.18987 1.16357 m
.20253 1.16357 L
.20253 1.18732 L
.18987 1.18732 L
F
.77848 1.53758 m
.77848 1.54352 L
.77532 1.54352 L
.77532 1.53758 L
F
.78682 1.54352 m
.78481 1.54352 L
.78481 1.53164 L
.7894 1.53164 L
F
.16456 1.16357 m
.17722 1.16357 L
.17722 1.18732 L
.16456 1.18732 L
F
.74367 1.53164 m
.74684 1.53164 L
.74684 1.53758 L
.74367 1.53758 L
F
.17722 1.16357 m
.18987 1.16357 L
.18987 1.18732 L
.17722 1.18732 L
F
.74367 1.53758 m
.74051 1.53758 L
.74051 1.53164 L
.74367 1.53164 L
F
.74367 1.52571 m
.74367 1.53164 L
.74051 1.53164 L
.74051 1.52571 L
F
.74367 1.53758 m
.74367 1.54352 L
.74051 1.54352 L
.74051 1.53758 L
F
.73734 1.53758 m
.73734 1.54352 L
.73418 1.54352 L
.73418 1.53758 L
F
.74367 1.53758 m
.74684 1.53758 L
.74684 1.54352 L
.74367 1.54352 L
F
.74367 1.52571 m
.74051 1.52571 L
.74051 1.51977 L
.74367 1.51977 L
F
.74367 1.52571 m
.74684 1.52571 L
.74684 1.53164 L
.74367 1.53164 L
F
.4019 1.33574 m
.4019 1.34167 L
.39873 1.34167 L
.39873 1.33574 L
F
.38924 1.33574 m
.39241 1.33574 L
.39241 1.34167 L
.38924 1.34167 L
F
.4019 1.33574 m
.39873 1.33574 L
.39873 1.3298 L
.4019 1.3298 L
F
.38924 1.33574 m
.38608 1.33574 L
.38608 1.3298 L
.38924 1.3298 L
F
.38924 1.33574 m
.38924 1.34167 L
.38608 1.34167 L
.38608 1.33574 L
F
.39557 1.32386 m
.39557 1.31793 L
.39873 1.31793 L
.39873 1.32386 L
F
.37342 .55804 m
.37342 .56991 L
.36709 .56991 L
.36709 .55804 L
F
.23418 .7955 m
.23418 .80738 L
.22785 .80738 L
.22785 .7955 L
F
.38291 .55804 m
.37975 .55804 L
.37975 .5521 L
.38291 .5521 L
F
.37342 .55804 m
.36709 .55804 L
.36709 .54617 L
.37342 .54617 L
F
.23418 .81925 m
.24051 .81925 L
.24051 .83112 L
.23418 .83112 L
F
.23418 .7955 m
.22785 .7955 L
.22785 .78363 L
.23418 .78363 L
F
.23418 .7955 m
.24051 .7955 L
.24051 .80738 L
.23418 .80738 L
F
.38924 1.31793 m
.39241 1.31793 L
.39241 1.32386 L
.38924 1.32386 L
F
.4019 1.31793 m
.40506 1.31793 L
.40506 1.32386 L
.4019 1.32386 L
F
.23418 1.05672 m
.24051 1.05672 L
.24051 1.06859 L
.23418 1.06859 L
F
.38608 1.32386 m
.38608 1.3298 L
.38291 1.3298 L
.38291 1.32386 L
F
.37342 1.31793 m
.36709 1.31793 L
.36709 1.30605 L
.37342 1.30605 L
F
.37342 1.31793 m
.37342 1.3298 L
.36709 1.3298 L
.36709 1.31793 L
F
.23418 1.08046 m
.22785 1.08046 L
.22785 1.06859 L
.23418 1.06859 L
F
.4557 .78363 m
.46835 .78363 L
.46835 .80738 L
.4557 .80738 L
F
.75316 .32651 m
.75316 .33245 L
.75 .33245 L
.75 .32651 L
F
.76582 .32651 m
.76582 .33245 L
.76266 .33245 L
.76266 .32651 L
F
.76899 .32058 m
.77215 .32058 L
.77215 .32651 L
.76899 .32651 L
F
.25316 1.23481 m
.26582 1.23481 L
.26582 1.25856 L
.25316 1.25856 L
F
.22785 1.23481 m
.24051 1.23481 L
.24051 1.25856 L
.22785 1.25856 L
F
.75633 .32058 m
.75949 .32058 L
.75949 .32651 L
.75633 .32651 L
F
.41139 .55804 m
.41139 .56991 L
.40506 .56991 L
.40506 .55804 L
F
.74367 .32058 m
.74684 .32058 L
.74684 .32651 L
.74367 .32651 L
F
.41139 .55804 m
.41772 .55804 L
.41772 .56991 L
.41139 .56991 L
F
.39557 .55804 m
.39241 .55804 L
.39241 .5521 L
.39557 .5521 L
F
.43038 .78363 m
.44304 .78363 L
.44304 .80738 L
.43038 .80738 L
F
.34177 1.40104 m
.35443 1.40104 L
.35443 1.42478 L
.34177 1.42478 L
F
.44304 .78363 m
.4557 .78363 L
.4557 .80738 L
.44304 .80738 L
F
.41139 .53429 m
.41772 .53429 L
.41772 .54617 L
.41139 .54617 L
F
.74051 .36807 m
.74684 .36807 L
.74684 .37994 L
.74051 .37994 L
F
.39873 .54023 m
.39873 .54617 L
.39557 .54617 L
.39557 .54023 L
F
.4019 .53429 m
.40506 .53429 L
.40506 .54023 L
.4019 .54023 L
F
.17722 .71239 m
.18987 .71239 L
.18987 .73614 L
.17722 .73614 L
F
.16456 .71239 m
.17722 .71239 L
.17722 .73614 L
.16456 .73614 L
F
.21519 .71239 m
.22785 .71239 L
.22785 .73614 L
.21519 .73614 L
F
.38608 .54023 m
.38608 .54617 L
.38291 .54617 L
.38291 .54023 L
F
.37342 .53429 m
.36709 .53429 L
.36709 .52242 L
.37342 .52242 L
F
.24051 .73614 m
.25316 .73614 L
.25316 .75989 L
.24051 .75989 L
F
.38924 .53429 m
.39241 .53429 L
.39241 .54023 L
.38924 .54023 L
F
.37342 .55804 m
.37975 .55804 L
.37975 .56991 L
.37342 .56991 L
F
.41139 .53429 m
.40506 .53429 L
.40506 .52242 L
.41139 .52242 L
F
.37342 .53429 m
.37342 .54617 L
.36709 .54617 L
.36709 .53429 L
F
.23418 1.08046 m
.23418 1.09233 L
.22785 1.09233 L
.22785 1.08046 L
F
.23418 1.08046 m
.24051 1.08046 L
.24051 1.09233 L
.23418 1.09233 L
F
.22785 .71239 m
.24051 .71239 L
.24051 .73614 L
.22785 .73614 L
F
.23418 1.05672 m
.22785 1.05672 L
.22785 1.04484 L
.23418 1.04484 L
F
.23418 .81925 m
.23418 .83112 L
.22785 .83112 L
.22785 .81925 L
F
.23418 1.05672 m
.23418 1.06859 L
.22785 1.06859 L
.22785 1.05672 L
F
.23418 .81925 m
.22785 .81925 L
.22785 .80738 L
.23418 .80738 L
F
.24051 .71239 m
.25316 .71239 L
.25316 .73614 L
.24051 .73614 L
F
.26582 .71239 m
.27848 .71239 L
.27848 .73614 L
.26582 .73614 L
F
.20253 .71239 m
.21519 .71239 L
.21519 .73614 L
.20253 .73614 L
F
.18987 .71239 m
.20253 .71239 L
.20253 .73614 L
.18987 .73614 L
F
.75316 .36807 m
.75316 .37994 L
.74684 .37994 L
.74684 .36807 L
F
.75316 .36807 m
.75949 .36807 L
.75949 .37994 L
.75316 .37994 L
F
.25316 .71239 m
.26582 .71239 L
.26582 .73614 L
.25316 .73614 L
F
.13924 1.06859 m
.1519 1.06859 L
.1519 1.09233 L
.13924 1.09233 L
F
.12658 1.06859 m
.13924 1.06859 L
.13924 1.09233 L
.12658 1.09233 L
F
.71976 .34432 m
.72152 .34432 L
.72152 .3562 L
.71984 .3562 L
F
.11392 1.06859 m
.12658 1.06859 L
.12658 1.09233 L
.11392 1.09233 L
F
.10127 1.06859 m
.11392 1.06859 L
.11392 1.09233 L
.10127 1.09233 L
F
.08861 1.06859 m
.10127 1.06859 L
.10127 1.09233 L
.08861 1.09233 L
F
.34177 .73614 m
.35443 .73614 L
.35443 .75989 L
.34177 .75989 L
F
.35443 1.40104 m
.36709 1.40104 L
.36709 1.42478 L
.35443 1.42478 L
F
.11392 .71239 m
.12658 .71239 L
.12658 .73614 L
.11392 .73614 L
F
.37975 1.40104 m
.39241 1.40104 L
.39241 1.42478 L
.37975 1.42478 L
F
.78481 .32911 m
.78481 .33245 L
.77848 .33245 L
.77848 .32061 L
F
.74051 .35026 m
.74051 .3562 L
.73734 .3562 L
.73734 .35026 L
F
.74367 .34432 m
.74684 .34432 L
.74684 .35026 L
.74367 .35026 L
F
.72785 .34432 m
.72785 .3562 L
.72152 .3562 L
.72152 .34432 L
F
.51899 .73614 m
.53165 .73614 L
.53165 .75989 L
.51899 .75989 L
F
.53165 .73614 m
.5443 .73614 L
.5443 .75989 L
.53165 .75989 L
F
.4557 .73614 m
.46835 .73614 L
.46835 .75989 L
.4557 .75989 L
F
.46835 .73614 m
.48101 .73614 L
.48101 .75989 L
.46835 .75989 L
F
.50633 .73614 m
.51899 .73614 L
.51899 .75989 L
.50633 .75989 L
F
.37975 .73614 m
.39241 .73614 L
.39241 .75989 L
.37975 .75989 L
F
.49367 .73614 m
.50633 .73614 L
.50633 .75989 L
.49367 .75989 L
F
.16456 1.06859 m
.17722 1.06859 L
.17722 1.09233 L
.16456 1.09233 L
F
.1519 1.06859 m
.16456 1.06859 L
.16456 1.09233 L
.1519 1.09233 L
F
.20253 1.06859 m
.21519 1.06859 L
.21519 1.09233 L
.20253 1.09233 L
F
.23418 1.06859 m
.24051 1.06859 L
.24051 1.08046 L
.23418 1.08046 L
F
.17722 1.06859 m
.18987 1.06859 L
.18987 1.09233 L
.17722 1.09233 L
F
.48101 .73614 m
.49367 .73614 L
.49367 .75989 L
.48101 .75989 L
F
.18987 1.06859 m
.20253 1.06859 L
.20253 1.09233 L
.18987 1.09233 L
F
.21519 1.23481 m
.22785 1.23481 L
.22785 1.25856 L
.21519 1.25856 L
F
.17722 1.23481 m
.18987 1.23481 L
.18987 1.25856 L
.17722 1.25856 L
F
.74051 1.49602 m
.74684 1.49602 L
.74684 1.5079 L
.74051 1.5079 L
F
.20253 1.23481 m
.21519 1.23481 L
.21519 1.25856 L
.20253 1.25856 L
F
.76899 .34432 m
.77215 .34432 L
.77215 .35026 L
.76899 .35026 L
F
.76266 .34432 m
.75949 .34432 L
.75949 .33839 L
.76266 .33839 L
F
.18987 1.23481 m
.20253 1.23481 L
.20253 1.25856 L
.18987 1.25856 L
F
.3038 1.23481 m
.31646 1.23481 L
.31646 1.25856 L
.3038 1.25856 L
F
.26582 1.23481 m
.27848 1.23481 L
.27848 1.25856 L
.26582 1.25856 L
F
.24051 1.23481 m
.25316 1.23481 L
.25316 1.25856 L
.24051 1.25856 L
F
.29114 1.23481 m
.3038 1.23481 L
.3038 1.25856 L
.29114 1.25856 L
F
.75316 1.49602 m
.75949 1.49602 L
.75949 1.5079 L
.75316 1.5079 L
F
.73418 1.49824 m
.73418 1.5079 L
.72785 1.5079 L
.72785 1.5036 L
F
.27848 1.23481 m
.29114 1.23481 L
.29114 1.25856 L
.27848 1.25856 L
F
.13924 .71239 m
.1519 .71239 L
.1519 .73614 L
.13924 .73614 L
F
.77848 .35026 m
.77848 .3562 L
.77532 .3562 L
.77532 .35026 L
F
.78771 .3562 m
.78481 .3562 L
.78481 .34432 L
.7894 .34432 L
F
.1519 .71239 m
.16456 .71239 L
.16456 .73614 L
.1519 .73614 L
F
.77532 .34432 m
.77215 .34432 L
.77215 .33839 L
.77532 .33839 L
F
.36709 1.40104 m
.37975 1.40104 L
.37975 1.42478 L
.36709 1.42478 L
F
.12658 .71239 m
.13924 .71239 L
.13924 .73614 L
.12658 .73614 L
F
.75633 .34432 m
.75949 .34432 L
.75949 .35026 L
.75633 .35026 L
F
.73734 .34432 m
.73418 .34432 L
.73418 .33839 L
.73734 .33839 L
F
.76582 .35026 m
.76582 .3562 L
.76266 .3562 L
.76266 .35026 L
F
.75316 .35026 m
.75316 .3562 L
.75 .3562 L
.75 .35026 L
F
.7894 .34432 m
.78481 .34432 L
.78481 .33245 L
.78682 .33245 L
F
.75 .34432 m
.74684 .34432 L
.74684 .33839 L
.75 .33839 L
F
.31646 1.23481 m
.32911 1.23481 L
.32911 1.25856 L
.31646 1.25856 L
F
.58228 1.13983 m
.59494 1.13983 L
.59494 1.16357 L
.58228 1.16357 L
F
.56962 1.13983 m
.58228 1.13983 L
.58228 1.16357 L
.56962 1.16357 L
F
.55696 1.13983 m
.56962 1.13983 L
.56962 1.16357 L
.55696 1.16357 L
F
.50633 1.13983 m
.51899 1.13983 L
.51899 1.16357 L
.50633 1.16357 L
F
.5443 1.13983 m
.55696 1.13983 L
.55696 1.16357 L
.5443 1.16357 L
F
.51899 1.13983 m
.53165 1.13983 L
.53165 1.16357 L
.51899 1.16357 L
F
.48101 1.13983 m
.49367 1.13983 L
.49367 1.16357 L
.48101 1.16357 L
F
.43038 1.13983 m
.44304 1.13983 L
.44304 1.16357 L
.43038 1.16357 L
F
.46835 1.13983 m
.48101 1.13983 L
.48101 1.16357 L
.46835 1.16357 L
F
.49367 1.13983 m
.50633 1.13983 L
.50633 1.16357 L
.49367 1.16357 L
F
.4557 1.13983 m
.46835 1.13983 L
.46835 1.16357 L
.4557 1.16357 L
F
.44304 1.13983 m
.4557 1.13983 L
.4557 1.16357 L
.44304 1.16357 L
F
.41772 1.13983 m
.43038 1.13983 L
.43038 1.16357 L
.41772 1.16357 L
F
.36709 1.13983 m
.37975 1.13983 L
.37975 1.16357 L
.36709 1.16357 L
F
.40506 1.13983 m
.41772 1.13983 L
.41772 1.16357 L
.40506 1.16357 L
F
.25316 1.13983 m
.26582 1.13983 L
.26582 1.16357 L
.25316 1.16357 L
F
.37975 1.13983 m
.39241 1.13983 L
.39241 1.16357 L
.37975 1.16357 L
F
.32911 1.13983 m
.34177 1.13983 L
.34177 1.16357 L
.32911 1.16357 L
F
.31646 1.13983 m
.32911 1.13983 L
.32911 1.16357 L
.31646 1.16357 L
F
.39241 1.13983 m
.40506 1.13983 L
.40506 1.16357 L
.39241 1.16357 L
F
.3038 1.13983 m
.31646 1.13983 L
.31646 1.16357 L
.3038 1.16357 L
F
.29114 1.13983 m
.3038 1.13983 L
.3038 1.16357 L
.29114 1.16357 L
F
.53165 1.13983 m
.5443 1.13983 L
.5443 1.16357 L
.53165 1.16357 L
F
.27848 1.13983 m
.29114 1.13983 L
.29114 1.16357 L
.27848 1.16357 L
F
.22785 1.13983 m
.24051 1.13983 L
.24051 1.16357 L
.22785 1.16357 L
F
.26582 1.13983 m
.27848 1.13983 L
.27848 1.16357 L
.26582 1.16357 L
F
.24051 1.13983 m
.25316 1.13983 L
.25316 1.16357 L
.24051 1.16357 L
F
.1519 1.13983 m
.16456 1.13983 L
.16456 1.16357 L
.1519 1.16357 L
F
.13924 1.13983 m
.1519 1.13983 L
.1519 1.16357 L
.13924 1.16357 L
F
.21519 1.13983 m
.22785 1.13983 L
.22785 1.16357 L
.21519 1.16357 L
F
.20253 1.13983 m
.21519 1.13983 L
.21519 1.16357 L
.20253 1.16357 L
F
.16456 1.13983 m
.17722 1.13983 L
.17722 1.16357 L
.16456 1.16357 L
F
.50633 1.23481 m
.51899 1.23481 L
.51899 1.25856 L
.50633 1.25856 L
F
.17722 1.13983 m
.18987 1.13983 L
.18987 1.16357 L
.17722 1.16357 L
F
.11392 1.13983 m
.12658 1.13983 L
.12658 1.16357 L
.11392 1.16357 L
F
.35443 1.23481 m
.36709 1.23481 L
.36709 1.25856 L
.35443 1.25856 L
F
.32911 1.23481 m
.34177 1.23481 L
.34177 1.25856 L
.32911 1.25856 L
F
.12658 1.13983 m
.13924 1.13983 L
.13924 1.16357 L
.12658 1.16357 L
F
.34177 1.23481 m
.35443 1.23481 L
.35443 1.25856 L
.34177 1.25856 L
F
.18987 1.13983 m
.20253 1.13983 L
.20253 1.16357 L
.18987 1.16357 L
F
.36709 1.23481 m
.37975 1.23481 L
.37975 1.25856 L
.36709 1.25856 L
F
.39241 1.23481 m
.40506 1.23481 L
.40506 1.25856 L
.39241 1.25856 L
F
.4557 1.23481 m
.46835 1.23481 L
.46835 1.25856 L
.4557 1.25856 L
F
.41772 1.23481 m
.43038 1.23481 L
.43038 1.25856 L
.41772 1.25856 L
F
.37975 1.23481 m
.39241 1.23481 L
.39241 1.25856 L
.37975 1.25856 L
F
.43038 1.23481 m
.44304 1.23481 L
.44304 1.25856 L
.43038 1.25856 L
F
.44304 1.23481 m
.4557 1.23481 L
.4557 1.25856 L
.44304 1.25856 L
F
.46835 1.23481 m
.48101 1.23481 L
.48101 1.25856 L
.46835 1.25856 L
F
.5443 1.23481 m
.55696 1.23481 L
.55696 1.25856 L
.5443 1.25856 L
F
.55696 1.23481 m
.56962 1.23481 L
.56962 1.25856 L
.55696 1.25856 L
F
.48101 1.23481 m
.49367 1.23481 L
.49367 1.25856 L
.48101 1.25856 L
F
.49367 1.23481 m
.50633 1.23481 L
.50633 1.25856 L
.49367 1.25856 L
F
.53165 1.23481 m
.5443 1.23481 L
.5443 1.25856 L
.53165 1.25856 L
F
.40506 1.23481 m
.41772 1.23481 L
.41772 1.25856 L
.40506 1.25856 L
F
.51899 1.23481 m
.53165 1.23481 L
.53165 1.25856 L
.51899 1.25856 L
F
.13924 .73614 m
.1519 .73614 L
.1519 .75989 L
.13924 .75989 L
F
.56962 1.23481 m
.58228 1.23481 L
.58228 1.25856 L
.56962 1.25856 L
F
.16456 .73614 m
.17722 .73614 L
.17722 .75989 L
.16456 .75989 L
F
.12658 .73614 m
.13924 .73614 L
.13924 .75989 L
.12658 .75989 L
F
.20253 .59366 m
.21519 .59366 L
.21519 .61741 L
.20253 .61741 L
F
.18987 .59681 m
.18987 .61741 L
.17722 .61741 L
.17722 .61219 L
F
.77215 1.49937 m
.77215 1.5079 L
.76582 1.5079 L
.76582 1.49613 L
F
.24051 1.11608 m
.25316 1.11608 L
.25316 1.13983 L
.24051 1.13983 L
F
.22785 1.11608 m
.24051 1.11608 L
.24051 1.13983 L
.22785 1.13983 L
F
.21519 1.11608 m
.22785 1.11608 L
.22785 1.13983 L
.21519 1.13983 L
F
.20253 1.11608 m
.21519 1.11608 L
.21519 1.13983 L
.20253 1.13983 L
F
.10127 .73614 m
.11392 .73614 L
.11392 .75989 L
.10127 .75989 L
F
.1519 .73614 m
.16456 .73614 L
.16456 .75989 L
.1519 .75989 L
F
.11392 .73614 m
.12658 .73614 L
.12658 .75989 L
.11392 .75989 L
F
.17722 .73614 m
.18987 .73614 L
.18987 .75989 L
.17722 .75989 L
F
.18987 .73614 m
.20253 .73614 L
.20253 .75989 L
.18987 .75989 L
F
.20253 .73614 m
.21519 .73614 L
.21519 .75989 L
.20253 .75989 L
F
.21519 .73614 m
.22785 .73614 L
.22785 .75989 L
.21519 .75989 L
F
.56962 1.11608 m
.58228 1.11608 L
.58228 1.13983 L
.56962 1.13983 L
F
.22785 .73614 m
.24051 .73614 L
.24051 .75989 L
.22785 .75989 L
F
.58228 1.11608 m
.59494 1.11608 L
.59494 1.13983 L
.58228 1.13983 L
F
.5443 1.11608 m
.55696 1.11608 L
.55696 1.13983 L
.5443 1.13983 L
F
.49367 1.11608 m
.50633 1.11608 L
.50633 1.13983 L
.49367 1.13983 L
F
.53165 1.11608 m
.5443 1.11608 L
.5443 1.13983 L
.53165 1.13983 L
F
.55696 1.11608 m
.56962 1.11608 L
.56962 1.13983 L
.55696 1.13983 L
F
.51899 1.11608 m
.53165 1.11608 L
.53165 1.13983 L
.51899 1.13983 L
F
.50633 1.11608 m
.51899 1.11608 L
.51899 1.13983 L
.50633 1.13983 L
F
.48101 1.11608 m
.49367 1.11608 L
.49367 1.13983 L
.48101 1.13983 L
F
.44304 1.11608 m
.4557 1.11608 L
.4557 1.13983 L
.44304 1.13983 L
F
.46835 1.11608 m
.48101 1.11608 L
.48101 1.13983 L
.46835 1.13983 L
F
.34177 1.11608 m
.35443 1.11608 L
.35443 1.13983 L
.34177 1.13983 L
F
.32911 1.11608 m
.34177 1.11608 L
.34177 1.13983 L
.32911 1.13983 L
F
.40506 1.11608 m
.41772 1.11608 L
.41772 1.13983 L
.40506 1.13983 L
F
.39241 1.11608 m
.40506 1.11608 L
.40506 1.13983 L
.39241 1.13983 L
F
.4557 1.11608 m
.46835 1.11608 L
.46835 1.13983 L
.4557 1.13983 L
F
.35443 1.11608 m
.36709 1.11608 L
.36709 1.13983 L
.35443 1.13983 L
F
.37975 1.11608 m
.39241 1.11608 L
.39241 1.13983 L
.37975 1.13983 L
F
.35443 1.13983 m
.36709 1.13983 L
.36709 1.16357 L
.35443 1.16357 L
F
.34177 1.13983 m
.35443 1.13983 L
.35443 1.16357 L
.34177 1.16357 L
F
.31646 1.11608 m
.32911 1.11608 L
.32911 1.13983 L
.31646 1.13983 L
F
.3038 1.11608 m
.31646 1.11608 L
.31646 1.13983 L
.3038 1.13983 L
F
.36709 1.11608 m
.37975 1.11608 L
.37975 1.13983 L
.36709 1.13983 L
F
.13924 1.11608 m
.1519 1.11608 L
.1519 1.13983 L
.13924 1.13983 L
F
.26582 1.11608 m
.27848 1.11608 L
.27848 1.13983 L
.26582 1.13983 L
F
.11392 1.11608 m
.12658 1.11608 L
.12658 1.13983 L
.11392 1.13983 L
F
.1519 1.11608 m
.16456 1.11608 L
.16456 1.13983 L
.1519 1.13983 L
F
.29114 1.11608 m
.3038 1.11608 L
.3038 1.13983 L
.29114 1.13983 L
F
.27848 1.11608 m
.29114 1.11608 L
.29114 1.13983 L
.27848 1.13983 L
F
.25316 1.11608 m
.26582 1.11608 L
.26582 1.13983 L
.25316 1.13983 L
F
.41772 1.11608 m
.43038 1.11608 L
.43038 1.13983 L
.41772 1.13983 L
F
.16456 1.11608 m
.17722 1.11608 L
.17722 1.13983 L
.16456 1.13983 L
F
.43038 1.11608 m
.44304 1.11608 L
.44304 1.13983 L
.43038 1.13983 L
F
.17722 1.11608 m
.18987 1.11608 L
.18987 1.13983 L
.17722 1.13983 L
F
.12658 1.11608 m
.13924 1.11608 L
.13924 1.13983 L
.12658 1.13983 L
F
.18987 1.11608 m
.20253 1.11608 L
.20253 1.13983 L
.18987 1.13983 L
F
.50633 1.25856 m
.51899 1.25856 L
.51899 1.28231 L
.50633 1.28231 L
F
.51899 1.25856 m
.53165 1.25856 L
.53165 1.28231 L
.51899 1.28231 L
F
.44304 .68865 m
.4557 .68865 L
.4557 .71239 L
.44304 .71239 L
F
.53165 1.25856 m
.5443 1.25856 L
.5443 1.28231 L
.53165 1.28231 L
F
.31646 .68865 m
.32911 .68865 L
.32911 .71239 L
.31646 .71239 L
F
.5443 1.25856 m
.55696 1.25856 L
.55696 1.28231 L
.5443 1.28231 L
F
.56962 1.25856 m
.58228 1.25856 L
.58228 1.28231 L
.56962 1.28231 L
F
.50633 .68865 m
.51899 .68865 L
.51899 .71239 L
.50633 .71239 L
F
.49367 .68865 m
.50633 .68865 L
.50633 .71239 L
.49367 .71239 L
F
.48101 .68865 m
.49367 .68865 L
.49367 .71239 L
.48101 .71239 L
F
.46835 .68865 m
.48101 .68865 L
.48101 .71239 L
.46835 .71239 L
F
.41772 .68865 m
.43038 .68865 L
.43038 .71239 L
.41772 .71239 L
F
.4557 .68865 m
.46835 .68865 L
.46835 .71239 L
.4557 .71239 L
F
.43038 .68865 m
.44304 .68865 L
.44304 .71239 L
.43038 .71239 L
F
.20253 .68865 m
.21519 .68865 L
.21519 .71239 L
.20253 .71239 L
F
.18987 .68865 m
.20253 .68865 L
.20253 .71239 L
.18987 .71239 L
F
.46835 1.25856 m
.48101 1.25856 L
.48101 1.28231 L
.46835 1.28231 L
F
.17722 .68865 m
.18987 .68865 L
.18987 .71239 L
.17722 .71239 L
F
.12658 .68865 m
.13924 .68865 L
.13924 .71239 L
.12658 .71239 L
F
.16456 .68865 m
.17722 .68865 L
.17722 .71239 L
.16456 .71239 L
F
.13924 .68865 m
.1519 .68865 L
.1519 .71239 L
.13924 .71239 L
F
.41772 1.25856 m
.43038 1.25856 L
.43038 1.28231 L
.41772 1.28231 L
F
.43038 1.25856 m
.44304 1.25856 L
.44304 1.28231 L
.43038 1.28231 L
F
.55696 1.25856 m
.56962 1.25856 L
.56962 1.28231 L
.55696 1.28231 L
F
.44304 1.25856 m
.4557 1.25856 L
.4557 1.28231 L
.44304 1.28231 L
F
.49367 1.25856 m
.50633 1.25856 L
.50633 1.28231 L
.49367 1.28231 L
F
.4557 1.25856 m
.46835 1.25856 L
.46835 1.28231 L
.4557 1.28231 L
F
.48101 1.25856 m
.49367 1.25856 L
.49367 1.28231 L
.48101 1.28231 L
F
.32911 .73614 m
.34177 .73614 L
.34177 .75989 L
.32911 .75989 L
F
.58228 .68865 m
.59494 .68865 L
.59494 .71239 L
.58228 .71239 L
F
.48101 1.09233 m
.49367 1.09233 L
.49367 1.11608 L
.48101 1.11608 L
F
.56962 .68865 m
.58228 .68865 L
.58228 .71239 L
.56962 .71239 L
F
.51899 .68865 m
.53165 .68865 L
.53165 .71239 L
.51899 .71239 L
F
.55696 .68865 m
.56962 .68865 L
.56962 .71239 L
.55696 .71239 L
F
.53165 .68865 m
.5443 .68865 L
.5443 .71239 L
.53165 .71239 L
F
.5443 1.09233 m
.55696 1.09233 L
.55696 1.11608 L
.5443 1.11608 L
F
.53165 1.09233 m
.5443 1.09233 L
.5443 1.11608 L
.53165 1.11608 L
F
.51899 1.09233 m
.53165 1.09233 L
.53165 1.11608 L
.51899 1.11608 L
F
.50633 1.09233 m
.51899 1.09233 L
.51899 1.11608 L
.50633 1.11608 L
F
.4557 1.09233 m
.46835 1.09233 L
.46835 1.11608 L
.4557 1.11608 L
F
.49367 1.09233 m
.50633 1.09233 L
.50633 1.11608 L
.49367 1.11608 L
F
.46835 1.09233 m
.48101 1.09233 L
.48101 1.11608 L
.46835 1.11608 L
F
.40506 .68865 m
.41772 .68865 L
.41772 .71239 L
.40506 .71239 L
F
.39241 .68865 m
.40506 .68865 L
.40506 .71239 L
.39241 .71239 L
F
.29114 .73614 m
.3038 .73614 L
.3038 .75989 L
.29114 .75989 L
F
.37975 .68865 m
.39241 .68865 L
.39241 .71239 L
.37975 .71239 L
F
.36709 .68865 m
.37975 .68865 L
.37975 .71239 L
.36709 .71239 L
F
.35443 .68865 m
.36709 .68865 L
.36709 .71239 L
.35443 .71239 L
F
.34177 .68865 m
.35443 .68865 L
.35443 .71239 L
.34177 .71239 L
F
.10127 1.11608 m
.11392 1.11608 L
.11392 1.13983 L
.10127 1.13983 L
F
.25316 .73614 m
.26582 .73614 L
.26582 .75989 L
.25316 .75989 L
F
.5443 .68865 m
.55696 .68865 L
.55696 .71239 L
.5443 .71239 L
F
.26582 .73614 m
.27848 .73614 L
.27848 .75989 L
.26582 .75989 L
F
.31646 .73614 m
.32911 .73614 L
.32911 .75989 L
.31646 .75989 L
F
.27848 .73614 m
.29114 .73614 L
.29114 .75989 L
.27848 .75989 L
F
.3038 .73614 m
.31646 .73614 L
.31646 .75989 L
.3038 .75989 L
F
.11392 1.09233 m
.12658 1.09233 L
.12658 1.11608 L
.11392 1.11608 L
F
.10127 1.09233 m
.11392 1.09233 L
.11392 1.11608 L
.10127 1.11608 L
F
.08861 1.09233 m
.10127 1.09233 L
.10127 1.11608 L
.08861 1.11608 L
F
.07841 1.09233 m
.08861 1.09233 L
.08861 1.11608 L
.08576 1.11608 L
F
.59904 .73614 m
.5991 .75989 L
.59494 .75989 L
.59494 .73614 L
F
.20253 1.18732 m
.21519 1.18732 L
.21519 1.21107 L
.20253 1.21107 L
F
.18987 1.18732 m
.20253 1.18732 L
.20253 1.21107 L
.18987 1.21107 L
F
.37975 1.25856 m
.39241 1.25856 L
.39241 1.28231 L
.37975 1.28231 L
F
.13924 1.09233 m
.1519 1.09233 L
.1519 1.11608 L
.13924 1.11608 L
F
.40506 1.25856 m
.41772 1.25856 L
.41772 1.28231 L
.40506 1.28231 L
F
.36709 1.25856 m
.37975 1.25856 L
.37975 1.28231 L
.36709 1.28231 L
F
.16456 1.09233 m
.17722 1.09233 L
.17722 1.11608 L
.16456 1.11608 L
F
.1519 1.09233 m
.16456 1.09233 L
.16456 1.11608 L
.1519 1.11608 L
F
.12658 1.09233 m
.13924 1.09233 L
.13924 1.11608 L
.12658 1.11608 L
F
.24051 1.18732 m
.25316 1.18732 L
.25316 1.21107 L
.24051 1.21107 L
F
.22785 1.18732 m
.24051 1.18732 L
.24051 1.21107 L
.22785 1.21107 L
F
.5991 .75989 m
.59911 .78363 L
.59494 .78363 L
.59494 .75989 L
F
.21519 1.18732 m
.22785 1.18732 L
.22785 1.21107 L
.21519 1.21107 L
F
.32911 1.21107 m
.34177 1.21107 L
.34177 1.23481 L
.32911 1.23481 L
F
.59135 .59366 m
.59465 .61741 L
.58228 .61741 L
.58228 .59366 L
F
.31646 1.21107 m
.32911 1.21107 L
.32911 1.23481 L
.31646 1.23481 L
F
.59888 .71239 m
.59904 .73614 L
.59494 .73614 L
.59494 .71239 L
F
.17722 1.18732 m
.18987 1.18732 L
.18987 1.21107 L
.17722 1.21107 L
F
.16456 1.18732 m
.17722 1.18732 L
.17722 1.21107 L
.16456 1.21107 L
F
.59854 .68865 m
.59888 .71239 L
.59494 .71239 L
.59494 .68865 L
F
.59911 .80738 m
.59911 .83112 L
.59494 .83112 L
.59494 .80738 L
F
.25316 1.18732 m
.26582 1.18732 L
.26582 1.21107 L
.25316 1.21107 L
F
.59911 .78363 m
.59911 .80738 L
.59494 .80738 L
.59494 .78363 L
F
.25316 1.09233 m
.26582 1.09233 L
.26582 1.11608 L
.25316 1.11608 L
F
.24051 1.09233 m
.25316 1.09233 L
.25316 1.11608 L
.24051 1.11608 L
F
.25316 .68865 m
.26582 .68865 L
.26582 .71239 L
.25316 .71239 L
F
.24051 .68865 m
.25316 .68865 L
.25316 .71239 L
.24051 .71239 L
F
.17722 1.09233 m
.18987 1.09233 L
.18987 1.11608 L
.17722 1.11608 L
F
.21519 1.09233 m
.22785 1.09233 L
.22785 1.11608 L
.21519 1.11608 L
F
.18987 1.09233 m
.20253 1.09233 L
.20253 1.11608 L
.18987 1.11608 L
F
.26582 .68865 m
.27848 .68865 L
.27848 .71239 L
.26582 .71239 L
F
.3038 .68865 m
.31646 .68865 L
.31646 .71239 L
.3038 .71239 L
F
.1519 .68865 m
.16456 .68865 L
.16456 .71239 L
.1519 .71239 L
F
.27848 .68865 m
.29114 .68865 L
.29114 .71239 L
.27848 .71239 L
F
.22785 .68865 m
.24051 .68865 L
.24051 .71239 L
.22785 .71239 L
F
.21519 .68865 m
.22785 .68865 L
.22785 .71239 L
.21519 .71239 L
F
.29114 .68865 m
.3038 .68865 L
.3038 .71239 L
.29114 .71239 L
F
.29114 1.09233 m
.3038 1.09233 L
.3038 1.11608 L
.29114 1.11608 L
F
.35443 1.25856 m
.36709 1.25856 L
.36709 1.28231 L
.35443 1.28231 L
F
.26582 1.09233 m
.27848 1.09233 L
.27848 1.11608 L
.26582 1.11608 L
F
.3038 1.09233 m
.31646 1.09233 L
.31646 1.11608 L
.3038 1.11608 L
F
.34177 1.25856 m
.35443 1.25856 L
.35443 1.28231 L
.34177 1.28231 L
F
.39241 1.25856 m
.40506 1.25856 L
.40506 1.28231 L
.39241 1.28231 L
F
.32911 1.25856 m
.34177 1.25856 L
.34177 1.28231 L
.32911 1.28231 L
F
.32911 1.09233 m
.34177 1.09233 L
.34177 1.11608 L
.32911 1.11608 L
F
.31646 1.09233 m
.32911 1.09233 L
.32911 1.11608 L
.31646 1.11608 L
F
.20253 1.09233 m
.21519 1.09233 L
.21519 1.11608 L
.20253 1.11608 L
F
.34177 1.09233 m
.35443 1.09233 L
.35443 1.11608 L
.34177 1.11608 L
F
.27848 1.09233 m
.29114 1.09233 L
.29114 1.11608 L
.27848 1.11608 L
F
.35443 1.09233 m
.36709 1.09233 L
.36709 1.11608 L
.35443 1.11608 L
F
.37975 1.04484 m
.39241 1.04484 L
.39241 1.06859 L
.37975 1.06859 L
F
.32911 1.04484 m
.34177 1.04484 L
.34177 1.06859 L
.32911 1.06859 L
F
.36709 1.04484 m
.37975 1.04484 L
.37975 1.06859 L
.36709 1.06859 L
F
.39241 1.04484 m
.40506 1.04484 L
.40506 1.06859 L
.39241 1.06859 L
F
.35443 1.04484 m
.36709 1.04484 L
.36709 1.06859 L
.35443 1.06859 L
F
.34177 1.04484 m
.35443 1.04484 L
.35443 1.06859 L
.34177 1.06859 L
F
.31646 1.04484 m
.32911 1.04484 L
.32911 1.06859 L
.31646 1.06859 L
F
.40506 1.06859 m
.41772 1.06859 L
.41772 1.09233 L
.40506 1.09233 L
F
.34177 1.06859 m
.35443 1.06859 L
.35443 1.09233 L
.34177 1.09233 L
F
.37975 1.06859 m
.39241 1.06859 L
.39241 1.09233 L
.37975 1.09233 L
F
.41772 1.06859 m
.43038 1.06859 L
.43038 1.09233 L
.41772 1.09233 L
F
.36709 1.06859 m
.37975 1.06859 L
.37975 1.09233 L
.36709 1.09233 L
F
.35443 1.06859 m
.36709 1.06859 L
.36709 1.09233 L
.35443 1.09233 L
F
.32911 1.06859 m
.34177 1.06859 L
.34177 1.09233 L
.32911 1.09233 L
F
.18987 1.04484 m
.20253 1.04484 L
.20253 1.06859 L
.18987 1.06859 L
F
.13924 1.04484 m
.1519 1.04484 L
.1519 1.06859 L
.13924 1.06859 L
F
.17722 1.04484 m
.18987 1.04484 L
.18987 1.06859 L
.17722 1.06859 L
F
.20253 1.04484 m
.21519 1.04484 L
.21519 1.06859 L
.20253 1.06859 L
F
.16456 1.04484 m
.17722 1.04484 L
.17722 1.06859 L
.16456 1.06859 L
F
.1519 1.04484 m
.16456 1.04484 L
.16456 1.06859 L
.1519 1.06859 L
F
.12658 1.04484 m
.13924 1.04484 L
.13924 1.06859 L
.12658 1.06859 L
F
.46835 1.04484 m
.48101 1.04484 L
.48101 1.06859 L
.46835 1.06859 L
F
.41772 1.04484 m
.43038 1.04484 L
.43038 1.06859 L
.41772 1.06859 L
F
.4557 1.04484 m
.46835 1.04484 L
.46835 1.06859 L
.4557 1.06859 L
F
.48101 1.04484 m
.49367 1.04484 L
.49367 1.06859 L
.48101 1.06859 L
F
.44304 1.04484 m
.4557 1.04484 L
.4557 1.06859 L
.44304 1.06859 L
F
.43038 1.04484 m
.44304 1.04484 L
.44304 1.06859 L
.43038 1.06859 L
F
.40506 1.04484 m
.41772 1.04484 L
.41772 1.06859 L
.40506 1.06859 L
F
.49367 1.06859 m
.50633 1.06859 L
.50633 1.09233 L
.49367 1.09233 L
F
.44304 1.06859 m
.4557 1.06859 L
.4557 1.09233 L
.44304 1.09233 L
F
.48101 1.06859 m
.49367 1.06859 L
.49367 1.09233 L
.48101 1.09233 L
F
.50633 1.06859 m
.51899 1.06859 L
.51899 1.09233 L
.50633 1.09233 L
F
.46835 1.06859 m
.48101 1.06859 L
.48101 1.09233 L
.46835 1.09233 L
F
.4557 1.06859 m
.46835 1.06859 L
.46835 1.09233 L
.4557 1.09233 L
F
.43038 1.06859 m
.44304 1.06859 L
.44304 1.09233 L
.43038 1.09233 L
F
.36709 .73614 m
.37975 .73614 L
.37975 .75989 L
.36709 .75989 L
F
.43038 .73614 m
.44304 .73614 L
.44304 .75989 L
.43038 .75989 L
F
.39241 .73614 m
.40506 .73614 L
.40506 .75989 L
.39241 .75989 L
F
.35443 .73614 m
.36709 .73614 L
.36709 .75989 L
.35443 .75989 L
F
.40506 .73614 m
.41772 .73614 L
.41772 .75989 L
.40506 .75989 L
F
.41772 .73614 m
.43038 .73614 L
.43038 .75989 L
.41772 .75989 L
F
.44304 .73614 m
.4557 .73614 L
.4557 .75989 L
.44304 .75989 L
F
.3038 1.06859 m
.31646 1.06859 L
.31646 1.09233 L
.3038 1.09233 L
F
.25316 1.06859 m
.26582 1.06859 L
.26582 1.09233 L
.25316 1.09233 L
F
.29114 1.06859 m
.3038 1.06859 L
.3038 1.09233 L
.29114 1.09233 L
F
.31646 1.06859 m
.32911 1.06859 L
.32911 1.09233 L
.31646 1.09233 L
F
.26582 1.06859 m
.27848 1.06859 L
.27848 1.09233 L
.26582 1.09233 L
F
.39241 1.06859 m
.40506 1.06859 L
.40506 1.09233 L
.39241 1.09233 L
F
.27848 1.06859 m
.29114 1.06859 L
.29114 1.09233 L
.27848 1.09233 L
F
.56962 1.06859 m
.58228 1.06859 L
.58228 1.09233 L
.56962 1.09233 L
F
.53165 1.06859 m
.5443 1.06859 L
.5443 1.09233 L
.53165 1.09233 L
F
.58228 1.06859 m
.59494 1.06859 L
.59494 1.09233 L
.58228 1.09233 L
F
.55696 1.06859 m
.56962 1.06859 L
.56962 1.09233 L
.55696 1.09233 L
F
.5443 1.06859 m
.55696 1.06859 L
.55696 1.09233 L
.5443 1.09233 L
F
.51899 1.06859 m
.53165 1.06859 L
.53165 1.09233 L
.51899 1.09233 L
F
.29114 1.25856 m
.3038 1.25856 L
.3038 1.28231 L
.29114 1.28231 L
F
.56962 1.09233 m
.58228 1.09233 L
.58228 1.11608 L
.56962 1.11608 L
F
.3038 1.25856 m
.31646 1.25856 L
.31646 1.28231 L
.3038 1.28231 L
F
.27848 1.25856 m
.29114 1.25856 L
.29114 1.28231 L
.27848 1.28231 L
F
.31646 1.25856 m
.32911 1.25856 L
.32911 1.28231 L
.31646 1.28231 L
F
.58228 1.09233 m
.59494 1.09233 L
.59494 1.11608 L
.58228 1.11608 L
F
.55696 1.09233 m
.56962 1.09233 L
.56962 1.11608 L
.55696 1.11608 L
F
.21519 1.28231 m
.22785 1.28231 L
.22785 1.30605 L
.21519 1.30605 L
F
.27848 1.28231 m
.29114 1.28231 L
.29114 1.30605 L
.27848 1.30605 L
F
.24051 1.28231 m
.25316 1.28231 L
.25316 1.30605 L
.24051 1.30605 L
F
.32911 .59366 m
.34177 .59366 L
.34177 .61741 L
.32911 .61741 L
F
.25316 1.28231 m
.26582 1.28231 L
.26582 1.30605 L
.25316 1.30605 L
F
.26582 1.28231 m
.27848 1.28231 L
.27848 1.30605 L
.26582 1.30605 L
F
.58228 1.04484 m
.59494 1.04484 L
.59494 1.06859 L
.58228 1.06859 L
F
.43038 1.09233 m
.44304 1.09233 L
.44304 1.11608 L
.43038 1.11608 L
F
.37975 1.09233 m
.39241 1.09233 L
.39241 1.11608 L
.37975 1.11608 L
F
.41772 1.09233 m
.43038 1.09233 L
.43038 1.11608 L
.41772 1.11608 L
F
.44304 1.09233 m
.4557 1.09233 L
.4557 1.11608 L
.44304 1.11608 L
F
.40506 1.09233 m
.41772 1.09233 L
.41772 1.11608 L
.40506 1.11608 L
F
.39241 1.09233 m
.40506 1.09233 L
.40506 1.11608 L
.39241 1.11608 L
F
.36709 1.09233 m
.37975 1.09233 L
.37975 1.11608 L
.36709 1.11608 L
F
.20253 1.25856 m
.21519 1.25856 L
.21519 1.28231 L
.20253 1.28231 L
F
.25316 1.25856 m
.26582 1.25856 L
.26582 1.28231 L
.25316 1.28231 L
F
.21519 1.25856 m
.22785 1.25856 L
.22785 1.28231 L
.21519 1.28231 L
F
.32911 .68865 m
.34177 .68865 L
.34177 .71239 L
.32911 .71239 L
F
.22785 1.25856 m
.24051 1.25856 L
.24051 1.28231 L
.22785 1.28231 L
F
.24051 1.25856 m
.25316 1.25856 L
.25316 1.28231 L
.24051 1.28231 L
F
.26582 1.25856 m
.27848 1.25856 L
.27848 1.28231 L
.26582 1.28231 L
F
.31646 .59366 m
.32911 .59366 L
.32911 .61741 L
.31646 .61741 L
F
.29114 .59366 m
.3038 .59366 L
.3038 .61741 L
.29114 .61741 L
F
.25316 .59366 m
.26582 .59366 L
.26582 .61741 L
.25316 .61741 L
F
.26582 .59366 m
.27848 .59366 L
.27848 .61741 L
.26582 .61741 L
F
.56962 .73614 m
.58228 .73614 L
.58228 .75989 L
.56962 .75989 L
F
.58228 .73614 m
.59494 .73614 L
.59494 .75989 L
.58228 .75989 L
F
.3038 .59366 m
.31646 .59366 L
.31646 .61741 L
.3038 .61741 L
F
.3038 1.04484 m
.31646 1.04484 L
.31646 1.06859 L
.3038 1.06859 L
F
.29114 1.04484 m
.3038 1.04484 L
.3038 1.06859 L
.29114 1.06859 L
F
.27848 1.04484 m
.29114 1.04484 L
.29114 1.06859 L
.27848 1.06859 L
F
.26582 1.04484 m
.27848 1.04484 L
.27848 1.06859 L
.26582 1.06859 L
F
.25316 1.04484 m
.26582 1.04484 L
.26582 1.06859 L
.25316 1.06859 L
F
.27848 .59366 m
.29114 .59366 L
.29114 .61741 L
.27848 .61741 L
F
.23418 1.04484 m
.24051 1.04484 L
.24051 1.05672 L
.23418 1.05672 L
F
.50633 1.04484 m
.51899 1.04484 L
.51899 1.06859 L
.50633 1.06859 L
F
.49367 1.04484 m
.50633 1.04484 L
.50633 1.06859 L
.49367 1.06859 L
F
.56962 1.04484 m
.58228 1.04484 L
.58228 1.06859 L
.56962 1.06859 L
F
.55696 1.04484 m
.56962 1.04484 L
.56962 1.06859 L
.55696 1.06859 L
F
.51899 1.04484 m
.53165 1.04484 L
.53165 1.06859 L
.51899 1.06859 L
F
.22785 1.28231 m
.24051 1.28231 L
.24051 1.30605 L
.22785 1.30605 L
F
.53165 1.04484 m
.5443 1.04484 L
.5443 1.06859 L
.53165 1.06859 L
F
.55696 .73614 m
.56962 .73614 L
.56962 .75989 L
.55696 .75989 L
F
.22785 .59366 m
.24051 .59366 L
.24051 .61741 L
.22785 .61741 L
F
.5443 .73614 m
.55696 .73614 L
.55696 .75989 L
.5443 .75989 L
F
.21519 .59366 m
.22785 .59366 L
.22785 .61741 L
.21519 .61741 L
F
.5443 1.04484 m
.55696 1.04484 L
.55696 1.06859 L
.5443 1.06859 L
F
.24051 .59366 m
.25316 .59366 L
.25316 .61741 L
.24051 .61741 L
F
.49367 1.0211 m
.50633 1.0211 L
.50633 1.04484 L
.49367 1.04484 L
F
.35443 1.28231 m
.36709 1.28231 L
.36709 1.30605 L
.35443 1.30605 L
F
.46835 1.0211 m
.48101 1.0211 L
.48101 1.04484 L
.46835 1.04484 L
F
.50633 1.0211 m
.51899 1.0211 L
.51899 1.04484 L
.50633 1.04484 L
F
.08861 1.04484 m
.10127 1.04484 L
.10127 1.06859 L
.08861 1.06859 L
F
.07595 1.04484 m
.08861 1.04484 L
.08861 1.06859 L
.07595 1.06859 L
F
.32911 1.28231 m
.34177 1.28231 L
.34177 1.30605 L
.32911 1.30605 L
F
.34177 1.28231 m
.35443 1.28231 L
.35443 1.30605 L
.34177 1.30605 L
F
.53165 1.0211 m
.5443 1.0211 L
.5443 1.04484 L
.53165 1.04484 L
F
.51899 1.0211 m
.53165 1.0211 L
.53165 1.04484 L
.51899 1.04484 L
F
.41772 1.0211 m
.43038 1.0211 L
.43038 1.04484 L
.41772 1.04484 L
F
.40506 1.0211 m
.41772 1.0211 L
.41772 1.04484 L
.40506 1.04484 L
F
.5443 1.0211 m
.55696 1.0211 L
.55696 1.04484 L
.5443 1.04484 L
F
.48101 1.0211 m
.49367 1.0211 L
.49367 1.04484 L
.48101 1.04484 L
F
.55696 1.0211 m
.56962 1.0211 L
.56962 1.04484 L
.55696 1.04484 L
F
.34177 .59366 m
.35443 .59366 L
.35443 .61741 L
.34177 .61741 L
F
.35443 .59366 m
.36709 .59366 L
.36709 .61741 L
.35443 .61741 L
F
.51899 1.28231 m
.53165 1.28231 L
.53165 1.30605 L
.51899 1.30605 L
F
.53165 1.28231 m
.5443 1.28231 L
.5443 1.30605 L
.53165 1.30605 L
F
.56962 1.28231 m
.58228 1.28231 L
.58228 1.30605 L
.56962 1.30605 L
F
.44304 1.28231 m
.4557 1.28231 L
.4557 1.30605 L
.44304 1.30605 L
F
.55696 1.28231 m
.56962 1.28231 L
.56962 1.30605 L
.55696 1.30605 L
F
.3038 1.28231 m
.31646 1.28231 L
.31646 1.30605 L
.3038 1.30605 L
F
.31646 1.28231 m
.32911 1.28231 L
.32911 1.30605 L
.31646 1.30605 L
F
.11392 1.04484 m
.12658 1.04484 L
.12658 1.06859 L
.11392 1.06859 L
F
.10127 1.04484 m
.11392 1.04484 L
.11392 1.06859 L
.10127 1.06859 L
F
.29114 1.28231 m
.3038 1.28231 L
.3038 1.30605 L
.29114 1.30605 L
F
.5443 1.28231 m
.55696 1.28231 L
.55696 1.30605 L
.5443 1.30605 L
F
.06876 1.04484 m
.07595 1.04484 L
.07595 1.06859 L
.07284 1.06859 L
F
.43038 .56991 m
.44304 .56991 L
.44304 .59366 L
.43038 .59366 L
F
.55696 .6649 m
.56962 .6649 L
.56962 .68865 L
.55696 .68865 L
F
.41772 .56991 m
.43038 .56991 L
.43038 .59366 L
.41772 .59366 L
F
.44304 .56991 m
.4557 .56991 L
.4557 .59366 L
.44304 .59366 L
F
.58228 1.0211 m
.59494 1.0211 L
.59494 1.04484 L
.58228 1.04484 L
F
.56962 1.0211 m
.58228 1.0211 L
.58228 1.04484 L
.56962 1.04484 L
F
.58228 .6649 m
.59494 .6649 L
.59494 .68865 L
.58228 .68865 L
F
.56962 .6649 m
.58228 .6649 L
.58228 .68865 L
.56962 .68865 L
F
.58228 .9736 m
.59494 .9736 L
.59494 .99735 L
.58228 .99735 L
F
.46835 .56991 m
.48101 .56991 L
.48101 .59366 L
.46835 .59366 L
F
.55696 .9736 m
.56962 .9736 L
.56962 .99735 L
.55696 .99735 L
F
.35443 .56991 m
.36709 .56991 L
.36709 .59366 L
.35443 .59366 L
F
.5443 1.30605 m
.55696 1.30605 L
.55696 1.3298 L
.5443 1.3298 L
F
.55696 1.30605 m
.56962 1.30605 L
.56962 1.3298 L
.55696 1.3298 L
F
.4557 .56991 m
.46835 .56991 L
.46835 .59366 L
.4557 .59366 L
F
.43038 1.0211 m
.44304 1.0211 L
.44304 1.04484 L
.43038 1.04484 L
F
.4557 1.0211 m
.46835 1.0211 L
.46835 1.04484 L
.4557 1.04484 L
F
.56962 .56991 m
.58228 .56991 L
.58228 .59366 L
.56962 .59366 L
F
.55696 .56991 m
.56962 .56991 L
.56962 .59366 L
.55696 .59366 L
F
.39241 1.0211 m
.40506 1.0211 L
.40506 1.04484 L
.39241 1.04484 L
F
.37975 1.0211 m
.39241 1.0211 L
.39241 1.04484 L
.37975 1.04484 L
F
.44304 1.0211 m
.4557 1.0211 L
.4557 1.04484 L
.44304 1.04484 L
F
.39241 .59366 m
.40506 .59366 L
.40506 .61741 L
.39241 .61741 L
F
.36709 .59366 m
.37975 .59366 L
.37975 .61741 L
.36709 .61741 L
F
.5443 .6649 m
.55696 .6649 L
.55696 .68865 L
.5443 .68865 L
F
.53165 .6649 m
.5443 .6649 L
.5443 .68865 L
.53165 .68865 L
F
.5443 .56991 m
.55696 .56991 L
.55696 .59366 L
.5443 .59366 L
F
.53165 .56991 m
.5443 .56991 L
.5443 .59366 L
.53165 .59366 L
F
.37975 .59366 m
.39241 .59366 L
.39241 .61741 L
.37975 .61741 L
F
.20253 1.0211 m
.21519 1.0211 L
.21519 1.04484 L
.20253 1.04484 L
F
.18987 1.0211 m
.20253 1.0211 L
.20253 1.04484 L
.18987 1.04484 L
F
.26582 1.0211 m
.27848 1.0211 L
.27848 1.04484 L
.26582 1.04484 L
F
.25316 1.0211 m
.26582 1.0211 L
.26582 1.04484 L
.25316 1.04484 L
F
.32911 1.0211 m
.34177 1.0211 L
.34177 1.04484 L
.32911 1.04484 L
F
.36709 1.0211 m
.37975 1.0211 L
.37975 1.04484 L
.36709 1.04484 L
F
.21519 1.0211 m
.22785 1.0211 L
.22785 1.04484 L
.21519 1.04484 L
F
.34177 1.0211 m
.35443 1.0211 L
.35443 1.04484 L
.34177 1.04484 L
F
.20253 .6649 m
.21519 .6649 L
.21519 .68865 L
.20253 .68865 L
F
.18987 .6649 m
.20253 .6649 L
.20253 .68865 L
.18987 .68865 L
F
.27848 .6649 m
.29114 .6649 L
.29114 .68865 L
.27848 .68865 L
F
.26582 .6649 m
.27848 .6649 L
.27848 .68865 L
.26582 .68865 L
F
.21519 .6649 m
.22785 .6649 L
.22785 .68865 L
.21519 .68865 L
F
.24051 1.0211 m
.25316 1.0211 L
.25316 1.04484 L
.24051 1.04484 L
F
.22785 .6649 m
.24051 .6649 L
.24051 .68865 L
.22785 .68865 L
F
.46835 .6649 m
.48101 .6649 L
.48101 .68865 L
.46835 .68865 L
F
.50633 .6649 m
.51899 .6649 L
.51899 .68865 L
.50633 .68865 L
F
.06581 1.0211 m
.07595 1.0211 L
.07595 1.04484 L
.06876 1.04484 L
F
.48101 .6649 m
.49367 .6649 L
.49367 .68865 L
.48101 .68865 L
F
.1519 1.0211 m
.16456 1.0211 L
.16456 1.04484 L
.1519 1.04484 L
F
.13924 1.0211 m
.1519 1.0211 L
.1519 1.04484 L
.13924 1.04484 L
F
.49367 .6649 m
.50633 .6649 L
.50633 .68865 L
.49367 .68865 L
F
.27848 1.0211 m
.29114 1.0211 L
.29114 1.04484 L
.27848 1.04484 L
F
.31646 1.0211 m
.32911 1.0211 L
.32911 1.04484 L
.31646 1.04484 L
F
.35443 1.0211 m
.36709 1.0211 L
.36709 1.04484 L
.35443 1.04484 L
F
.29114 1.0211 m
.3038 1.0211 L
.3038 1.04484 L
.29114 1.04484 L
F
.51899 .6649 m
.53165 .6649 L
.53165 .68865 L
.51899 .68865 L
F
.07595 1.0211 m
.08861 1.0211 L
.08861 1.04484 L
.07595 1.04484 L
F
.3038 1.0211 m
.31646 1.0211 L
.31646 1.04484 L
.3038 1.04484 L
F
.35443 .6649 m
.36709 .6649 L
.36709 .68865 L
.35443 .68865 L
F
.3038 .6649 m
.31646 .6649 L
.31646 .68865 L
.3038 .68865 L
F
.34177 .6649 m
.35443 .6649 L
.35443 .68865 L
.34177 .68865 L
F
.36709 .6649 m
.37975 .6649 L
.37975 .68865 L
.36709 .68865 L
F
.32911 .6649 m
.34177 .6649 L
.34177 .68865 L
.32911 .68865 L
F
.31646 .6649 m
.32911 .6649 L
.32911 .68865 L
.31646 .68865 L
F
.29114 .6649 m
.3038 .6649 L
.3038 .68865 L
.29114 .68865 L
F
.43038 1.28231 m
.44304 1.28231 L
.44304 1.30605 L
.43038 1.30605 L
F
.49367 1.28231 m
.50633 1.28231 L
.50633 1.30605 L
.49367 1.30605 L
F
.4557 1.28231 m
.46835 1.28231 L
.46835 1.30605 L
.4557 1.30605 L
F
.41772 1.28231 m
.43038 1.28231 L
.43038 1.30605 L
.41772 1.30605 L
F
.46835 1.28231 m
.48101 1.28231 L
.48101 1.30605 L
.46835 1.30605 L
F
.48101 1.28231 m
.49367 1.28231 L
.49367 1.30605 L
.48101 1.30605 L
F
.50633 1.28231 m
.51899 1.28231 L
.51899 1.30605 L
.50633 1.30605 L
F
.16456 .6649 m
.17722 .6649 L
.17722 .68865 L
.16456 .68865 L
F
.51899 .56991 m
.53165 .56991 L
.53165 .59366 L
.51899 .59366 L
F
.1519 .6649 m
.16456 .6649 L
.16456 .68865 L
.1519 .68865 L
F
.17722 .6649 m
.18987 .6649 L
.18987 .68865 L
.17722 .68865 L
F
.25316 .6649 m
.26582 .6649 L
.26582 .68865 L
.25316 .68865 L
F
.24051 .6649 m
.25316 .6649 L
.25316 .68865 L
.24051 .68865 L
F
.13924 .6649 m
.1519 .6649 L
.1519 .68865 L
.13924 .68865 L
F
.44304 .6649 m
.4557 .6649 L
.4557 .68865 L
.44304 .68865 L
F
.39241 .6649 m
.40506 .6649 L
.40506 .68865 L
.39241 .68865 L
F
.43038 .6649 m
.44304 .6649 L
.44304 .68865 L
.43038 .68865 L
F
.4557 .6649 m
.46835 .6649 L
.46835 .68865 L
.4557 .68865 L
F
.41772 .6649 m
.43038 .6649 L
.43038 .68865 L
.41772 .68865 L
F
.40506 .6649 m
.41772 .6649 L
.41772 .68865 L
.40506 .68865 L
F
.37975 .6649 m
.39241 .6649 L
.39241 .68865 L
.37975 .68865 L
F
.26582 1.30605 m
.27848 1.30605 L
.27848 1.3298 L
.26582 1.3298 L
F
.27848 1.30605 m
.29114 1.30605 L
.29114 1.3298 L
.27848 1.3298 L
F
.24051 .99735 m
.25316 .99735 L
.25316 1.0211 L
.24051 1.0211 L
F
.29114 1.30605 m
.3038 1.30605 L
.3038 1.3298 L
.29114 1.3298 L
F
.3038 1.30605 m
.31646 1.30605 L
.31646 1.3298 L
.3038 1.3298 L
F
.31646 1.30605 m
.32911 1.30605 L
.32911 1.3298 L
.31646 1.3298 L
F
.32911 1.30605 m
.34177 1.30605 L
.34177 1.3298 L
.32911 1.3298 L
F
.34177 1.30605 m
.35443 1.30605 L
.35443 1.3298 L
.34177 1.3298 L
F
.29114 .99735 m
.3038 .99735 L
.3038 1.0211 L
.29114 1.0211 L
F
.27848 .99735 m
.29114 .99735 L
.29114 1.0211 L
.27848 1.0211 L
F
.1519 .99735 m
.16456 .99735 L
.16456 1.0211 L
.1519 1.0211 L
F
.26582 .99735 m
.27848 .99735 L
.27848 1.0211 L
.26582 1.0211 L
F
.21519 .99735 m
.22785 .99735 L
.22785 1.0211 L
.21519 1.0211 L
F
.25316 .99735 m
.26582 .99735 L
.26582 1.0211 L
.25316 1.0211 L
F
.22785 .99735 m
.24051 .99735 L
.24051 1.0211 L
.22785 1.0211 L
F
.51899 .99735 m
.53165 .99735 L
.53165 1.0211 L
.51899 1.0211 L
F
.50633 .99735 m
.51899 .99735 L
.51899 1.0211 L
.50633 1.0211 L
F
.48101 .56991 m
.49367 .56991 L
.49367 .59366 L
.48101 .59366 L
F
.49367 .99735 m
.50633 .99735 L
.50633 1.0211 L
.49367 1.0211 L
F
.44304 .99735 m
.4557 .99735 L
.4557 1.0211 L
.44304 1.0211 L
F
.48101 .99735 m
.49367 .99735 L
.49367 1.0211 L
.48101 1.0211 L
F
.4557 .99735 m
.46835 .99735 L
.46835 1.0211 L
.4557 1.0211 L
F
.11392 .99735 m
.12658 .99735 L
.12658 1.0211 L
.11392 1.0211 L
F
.10127 .99735 m
.11392 .99735 L
.11392 1.0211 L
.10127 1.0211 L
F
.08861 .99735 m
.10127 .99735 L
.10127 1.0211 L
.08861 1.0211 L
F
.07595 .99735 m
.08861 .99735 L
.08861 1.0211 L
.07595 1.0211 L
F
.25316 1.30605 m
.26582 1.30605 L
.26582 1.3298 L
.25316 1.3298 L
F
.06382 .99735 m
.07595 .99735 L
.07595 1.0211 L
.06581 1.0211 L
F
.24051 1.30605 m
.25316 1.30605 L
.25316 1.3298 L
.24051 1.3298 L
F
.34177 .83112 m
.35443 .83112 L
.35443 .85487 L
.34177 .85487 L
F
.32911 .83112 m
.34177 .83112 L
.34177 .85487 L
.32911 .85487 L
F
.74367 1.54352 m
.74684 1.54352 L
.74684 1.54945 L
.74367 1.54945 L
F
.31646 .83112 m
.32911 .83112 L
.32911 .85487 L
.31646 .85487 L
F
.3038 .83112 m
.31646 .83112 L
.31646 .85487 L
.3038 .85487 L
F
.29114 .83112 m
.3038 .83112 L
.3038 .85487 L
.29114 .85487 L
F
.27848 .83112 m
.29114 .83112 L
.29114 .85487 L
.27848 .85487 L
F
.26582 .83112 m
.27848 .83112 L
.27848 .85487 L
.26582 .85487 L
F
.44304 1.37729 m
.4557 1.37729 L
.4557 1.40104 L
.44304 1.40104 L
F
.58228 .83112 m
.59494 .83112 L
.59494 .85487 L
.58228 .85487 L
F
.56962 .83112 m
.58228 .83112 L
.58228 .85487 L
.56962 .85487 L
F
.76899 1.54352 m
.77215 1.54352 L
.77215 1.54945 L
.76899 1.54945 L
F
.5443 .83112 m
.55696 .83112 L
.55696 .85487 L
.5443 .85487 L
F
.75633 1.54352 m
.75949 1.54352 L
.75949 1.54945 L
.75633 1.54945 L
F
.55696 .83112 m
.56962 .83112 L
.56962 .85487 L
.55696 .85487 L
F
.20253 .99735 m
.21519 .99735 L
.21519 1.0211 L
.20253 1.0211 L
F
.18987 .99735 m
.20253 .99735 L
.20253 1.0211 L
.18987 1.0211 L
F
.37975 .83112 m
.39241 .83112 L
.39241 .85487 L
.37975 .85487 L
F
.17722 .99735 m
.18987 .99735 L
.18987 1.0211 L
.17722 1.0211 L
F
.12658 .99735 m
.13924 .99735 L
.13924 1.0211 L
.12658 1.0211 L
F
.16456 .99735 m
.17722 .99735 L
.17722 1.0211 L
.16456 1.0211 L
F
.13924 .99735 m
.1519 .99735 L
.1519 1.0211 L
.13924 1.0211 L
F
.44304 .83112 m
.4557 .83112 L
.4557 .85487 L
.44304 .85487 L
F
.43038 .83112 m
.44304 .83112 L
.44304 .85487 L
.43038 .85487 L
F
.41772 .83112 m
.43038 .83112 L
.43038 .85487 L
.41772 .85487 L
F
.40506 .83112 m
.41772 .83112 L
.41772 .85487 L
.40506 .85487 L
F
.35443 .83112 m
.36709 .83112 L
.36709 .85487 L
.35443 .85487 L
F
.39241 .83112 m
.40506 .83112 L
.40506 .85487 L
.39241 .85487 L
F
.36709 .83112 m
.37975 .83112 L
.37975 .85487 L
.36709 .85487 L
F
.29114 .9736 m
.3038 .9736 L
.3038 .99735 L
.29114 .99735 L
F
.41772 .9736 m
.43038 .9736 L
.43038 .99735 L
.41772 .99735 L
F
.26582 .9736 m
.27848 .9736 L
.27848 .99735 L
.26582 .99735 L
F
.3038 .9736 m
.31646 .9736 L
.31646 .99735 L
.3038 .99735 L
F
.36709 .9736 m
.37975 .9736 L
.37975 .99735 L
.36709 .99735 L
F
.35443 .9736 m
.36709 .9736 L
.36709 .99735 L
.35443 .99735 L
F
.40506 .9736 m
.41772 .9736 L
.41772 .99735 L
.40506 .99735 L
F
.76899 .33245 m
.77215 .33245 L
.77215 .33839 L
.76899 .33839 L
F
.49367 .9736 m
.50633 .9736 L
.50633 .99735 L
.49367 .99735 L
F
.31646 .9736 m
.32911 .9736 L
.32911 .99735 L
.31646 .99735 L
F
.46835 .9736 m
.48101 .9736 L
.48101 .99735 L
.46835 .99735 L
F
.50633 .9736 m
.51899 .9736 L
.51899 .99735 L
.50633 .99735 L
F
.32911 .9736 m
.34177 .9736 L
.34177 .99735 L
.32911 .99735 L
F
.27848 .9736 m
.29114 .9736 L
.29114 .99735 L
.27848 .99735 L
F
.34177 .9736 m
.35443 .9736 L
.35443 .99735 L
.34177 .99735 L
F
.37342 1.30605 m
.37975 1.30605 L
.37975 1.31793 L
.37342 1.31793 L
F
.43038 1.30605 m
.44304 1.30605 L
.44304 1.3298 L
.43038 1.3298 L
F
.49367 1.30605 m
.50633 1.30605 L
.50633 1.3298 L
.49367 1.3298 L
F
.50633 1.30605 m
.51899 1.30605 L
.51899 1.3298 L
.50633 1.3298 L
F
.41139 1.30605 m
.41772 1.30605 L
.41772 1.31793 L
.41139 1.31793 L
F
.56962 .9736 m
.58228 .9736 L
.58228 .99735 L
.56962 .99735 L
F
.44304 1.30605 m
.4557 1.30605 L
.4557 1.3298 L
.44304 1.3298 L
F
.48101 1.30605 m
.49367 1.30605 L
.49367 1.3298 L
.48101 1.3298 L
F
.4557 1.30605 m
.46835 1.30605 L
.46835 1.3298 L
.4557 1.3298 L
F
.39241 .9736 m
.40506 .9736 L
.40506 .99735 L
.39241 .99735 L
F
.37975 .9736 m
.39241 .9736 L
.39241 .99735 L
.37975 .99735 L
F
.51899 1.30605 m
.53165 1.30605 L
.53165 1.3298 L
.51899 1.3298 L
F
.53165 1.30605 m
.5443 1.30605 L
.5443 1.3298 L
.53165 1.3298 L
F
.46835 1.30605 m
.48101 1.30605 L
.48101 1.3298 L
.46835 1.3298 L
F
.31646 .99735 m
.32911 .99735 L
.32911 1.0211 L
.31646 1.0211 L
F
.3038 .99735 m
.31646 .99735 L
.31646 1.0211 L
.3038 1.0211 L
F
.55696 .99735 m
.56962 .99735 L
.56962 1.0211 L
.55696 1.0211 L
F
.35443 .99735 m
.36709 .99735 L
.36709 1.0211 L
.35443 1.0211 L
F
.32911 .99735 m
.34177 .99735 L
.34177 1.0211 L
.32911 1.0211 L
F
.40506 .99735 m
.41772 .99735 L
.41772 1.0211 L
.40506 1.0211 L
F
.34177 .99735 m
.35443 .99735 L
.35443 1.0211 L
.34177 1.0211 L
F
.58228 .99735 m
.59494 .99735 L
.59494 1.0211 L
.58228 1.0211 L
F
.50633 .56991 m
.51899 .56991 L
.51899 .59366 L
.50633 .59366 L
F
.46835 .99735 m
.48101 .99735 L
.48101 1.0211 L
.46835 1.0211 L
F
.49367 .56991 m
.50633 .56991 L
.50633 .59366 L
.49367 .59366 L
F
.53165 .99735 m
.5443 .99735 L
.5443 1.0211 L
.53165 1.0211 L
F
.56962 .99735 m
.58228 .99735 L
.58228 1.0211 L
.56962 1.0211 L
F
.5443 .99735 m
.55696 .99735 L
.55696 1.0211 L
.5443 1.0211 L
F
.75633 .33245 m
.75949 .33245 L
.75949 .33839 L
.75633 .33839 L
F
.51899 .9736 m
.53165 .9736 L
.53165 .99735 L
.51899 .99735 L
F
.44304 .9736 m
.4557 .9736 L
.4557 .99735 L
.44304 .99735 L
F
.74367 .33245 m
.74684 .33245 L
.74684 .33839 L
.74367 .33839 L
F
.53165 .9736 m
.5443 .9736 L
.5443 .99735 L
.53165 .99735 L
F
.48101 .9736 m
.49367 .9736 L
.49367 .99735 L
.48101 .99735 L
F
.5443 .9736 m
.55696 .9736 L
.55696 .99735 L
.5443 .99735 L
F
.41772 .99735 m
.43038 .99735 L
.43038 1.0211 L
.41772 1.0211 L
F
.37975 .99735 m
.39241 .99735 L
.39241 1.0211 L
.37975 1.0211 L
F
.39241 .99735 m
.40506 .99735 L
.40506 1.0211 L
.39241 1.0211 L
F
.43038 .99735 m
.44304 .99735 L
.44304 1.0211 L
.43038 1.0211 L
F
.43038 .9736 m
.44304 .9736 L
.44304 .99735 L
.43038 .99735 L
F
.4557 .9736 m
.46835 .9736 L
.46835 .99735 L
.4557 .99735 L
F
.36709 .99735 m
.37975 .99735 L
.37975 1.0211 L
.36709 1.0211 L
F
.07595 .9736 m
.08861 .9736 L
.08861 .99735 L
.07595 .99735 L
F
.11392 .9736 m
.12658 .9736 L
.12658 .99735 L
.11392 .99735 L
F
.13924 .9736 m
.1519 .9736 L
.1519 .99735 L
.13924 .99735 L
F
.08861 .9736 m
.10127 .9736 L
.10127 .99735 L
.08861 .99735 L
F
.21519 .9736 m
.22785 .9736 L
.22785 .99735 L
.21519 .99735 L
F
.25316 .9736 m
.26582 .9736 L
.26582 .99735 L
.25316 .99735 L
F
.10127 .9736 m
.11392 .9736 L
.11392 .99735 L
.10127 .99735 L
F
.22785 .9736 m
.24051 .9736 L
.24051 .99735 L
.22785 .99735 L
F
.48101 .64115 m
.49367 .64115 L
.49367 .6649 L
.48101 .6649 L
F
.51899 .64115 m
.53165 .64115 L
.53165 .6649 L
.51899 .6649 L
F
.55696 .64115 m
.56962 .64115 L
.56962 .6649 L
.55696 .6649 L
F
.49367 .64115 m
.50633 .64115 L
.50633 .6649 L
.49367 .6649 L
F
.12658 .9736 m
.13924 .9736 L
.13924 .99735 L
.12658 .99735 L
F
.1519 .9736 m
.16456 .9736 L
.16456 .99735 L
.1519 .99735 L
F
.50633 .64115 m
.51899 .64115 L
.51899 .6649 L
.50633 .6649 L
F
.18987 .75989 m
.20253 .75989 L
.20253 .78363 L
.18987 .78363 L
F
.1519 .75989 m
.16456 .75989 L
.16456 .78363 L
.1519 .78363 L
F
.12658 .75989 m
.13924 .75989 L
.13924 .78363 L
.12658 .78363 L
F
.17722 .75989 m
.18987 .75989 L
.18987 .78363 L
.17722 .78363 L
F
.32911 .56991 m
.34177 .56991 L
.34177 .59366 L
.32911 .59366 L
F
.31646 .56991 m
.32911 .56991 L
.32911 .59366 L
.31646 .59366 L
F
.16456 .75989 m
.17722 .75989 L
.17722 .78363 L
.16456 .78363 L
F
.16456 .9736 m
.17722 .9736 L
.17722 .99735 L
.16456 .99735 L
F
.20253 .9736 m
.21519 .9736 L
.21519 .99735 L
.20253 .99735 L
F
.24051 .9736 m
.25316 .9736 L
.25316 .99735 L
.24051 .99735 L
F
.17722 .9736 m
.18987 .9736 L
.18987 .99735 L
.17722 .99735 L
F
.13924 .75989 m
.1519 .75989 L
.1519 .78363 L
.13924 .78363 L
F
.11392 .75989 m
.12658 .75989 L
.12658 .78363 L
.11392 .78363 L
F
.18987 .9736 m
.20253 .9736 L
.20253 .99735 L
.18987 .99735 L
F
.35443 .75989 m
.36709 .75989 L
.36709 .78363 L
.35443 .78363 L
F
.58228 .64115 m
.59494 .64115 L
.59494 .6649 L
.58228 .6649 L
F
.29114 .75989 m
.3038 .75989 L
.3038 .78363 L
.29114 .78363 L
F
.3038 .75989 m
.31646 .75989 L
.31646 .78363 L
.3038 .78363 L
F
.20253 .75989 m
.21519 .75989 L
.21519 .78363 L
.20253 .78363 L
F
.21519 .75989 m
.22785 .75989 L
.22785 .78363 L
.21519 .78363 L
F
.32911 .75989 m
.34177 .75989 L
.34177 .78363 L
.32911 .78363 L
F
.34177 .75989 m
.35443 .75989 L
.35443 .78363 L
.34177 .78363 L
F
.16456 .64115 m
.17722 .64115 L
.17722 .6649 L
.16456 .6649 L
F
.37975 .75989 m
.39241 .75989 L
.39241 .78363 L
.37975 .78363 L
F
.1519 .64342 m
.1519 .6649 L
.13924 .6649 L
.13924 .66036 L
F
.17722 .64115 m
.18987 .64115 L
.18987 .6649 L
.17722 .6649 L
F
.36709 .75989 m
.37975 .75989 L
.37975 .78363 L
.36709 .78363 L
F
.31646 .75989 m
.32911 .75989 L
.32911 .78363 L
.31646 .78363 L
F
.39241 .75989 m
.40506 .75989 L
.40506 .78363 L
.39241 .78363 L
F
.55696 .94986 m
.56962 .94986 L
.56962 .9736 L
.55696 .9736 L
F
.5443 .94986 m
.55696 .94986 L
.55696 .9736 L
.5443 .9736 L
F
.53165 .94986 m
.5443 .94986 L
.5443 .9736 L
.53165 .9736 L
F
.51899 .94986 m
.53165 .94986 L
.53165 .9736 L
.51899 .9736 L
F
.53165 .64115 m
.5443 .64115 L
.5443 .6649 L
.53165 .6649 L
F
.56962 .64115 m
.58228 .64115 L
.58228 .6649 L
.56962 .6649 L
F
.5443 .64115 m
.55696 .64115 L
.55696 .6649 L
.5443 .6649 L
F
.24051 .75989 m
.25316 .75989 L
.25316 .78363 L
.24051 .78363 L
F
.25316 .75989 m
.26582 .75989 L
.26582 .78363 L
.25316 .78363 L
F
.26582 .75989 m
.27848 .75989 L
.27848 .78363 L
.26582 .78363 L
F
.27848 .75989 m
.29114 .75989 L
.29114 .78363 L
.27848 .78363 L
F
.46835 .64115 m
.48101 .64115 L
.48101 .6649 L
.46835 .6649 L
F
.58228 .94986 m
.59494 .94986 L
.59494 .9736 L
.58228 .9736 L
F
.56962 .94986 m
.58228 .94986 L
.58228 .9736 L
.56962 .9736 L
F
.44304 .94986 m
.4557 .94986 L
.4557 .9736 L
.44304 .9736 L
F
.43038 .94986 m
.44304 .94986 L
.44304 .9736 L
.43038 .9736 L
F
.41772 .94986 m
.43038 .94986 L
.43038 .9736 L
.41772 .9736 L
F
.40506 .94986 m
.41772 .94986 L
.41772 .9736 L
.40506 .9736 L
F
.49367 .94986 m
.50633 .94986 L
.50633 .9736 L
.49367 .9736 L
F
.26582 .94986 m
.27848 .94986 L
.27848 .9736 L
.26582 .9736 L
F
.72785 .3562 m
.73418 .3562 L
.73418 .36807 L
.72785 .36807 L
F
.50633 .94986 m
.51899 .94986 L
.51899 .9736 L
.50633 .9736 L
F
.39241 .64115 m
.40506 .64115 L
.40506 .6649 L
.39241 .6649 L
F
.25316 1.3298 m
.26582 1.3298 L
.26582 1.35355 L
.25316 1.35355 L
F
.26582 1.3298 m
.27848 1.3298 L
.27848 1.35355 L
.26582 1.35355 L
F
.27848 1.3298 m
.29114 1.3298 L
.29114 1.35355 L
.27848 1.35355 L
F
.48101 .94986 m
.49367 .94986 L
.49367 .9736 L
.48101 .9736 L
F
.46835 .94986 m
.48101 .94986 L
.48101 .9736 L
.46835 .9736 L
F
.4557 .94986 m
.46835 .94986 L
.46835 .9736 L
.4557 .9736 L
F
.37975 .94986 m
.39241 .94986 L
.39241 .9736 L
.37975 .9736 L
F
.31646 .94986 m
.32911 .94986 L
.32911 .9736 L
.31646 .9736 L
F
.35443 .94986 m
.36709 .94986 L
.36709 .9736 L
.35443 .9736 L
F
.39241 .94986 m
.40506 .94986 L
.40506 .9736 L
.39241 .9736 L
F
.34177 .94986 m
.35443 .94986 L
.35443 .9736 L
.34177 .9736 L
F
.32911 .94986 m
.34177 .94986 L
.34177 .9736 L
.32911 .9736 L
F
.3038 .94986 m
.31646 .94986 L
.31646 .9736 L
.3038 .9736 L
F
.22785 .94986 m
.24051 .94986 L
.24051 .9736 L
.22785 .9736 L
F
.21519 .94986 m
.22785 .94986 L
.22785 .9736 L
.21519 .9736 L
F
.29114 .94986 m
.3038 .94986 L
.3038 .9736 L
.29114 .9736 L
F
.27848 .94986 m
.29114 .94986 L
.29114 .9736 L
.27848 .9736 L
F
.24051 .94986 m
.25316 .94986 L
.25316 .9736 L
.24051 .9736 L
F
.36709 .94986 m
.37975 .94986 L
.37975 .9736 L
.36709 .9736 L
F
.25316 .94986 m
.26582 .94986 L
.26582 .9736 L
.25316 .9736 L
F
.44304 .64115 m
.4557 .64115 L
.4557 .6649 L
.44304 .6649 L
F
.06329 .94986 m
.07595 .94986 L
.07595 .9736 L
.06329 .9736 L
F
.08861 .94986 m
.10127 .94986 L
.10127 .9736 L
.08861 .9736 L
F
.4557 .64115 m
.46835 .64115 L
.46835 .6649 L
.4557 .6649 L
F
.17722 .94986 m
.18987 .94986 L
.18987 .9736 L
.17722 .9736 L
F
.16456 .94986 m
.17722 .94986 L
.17722 .9736 L
.16456 .9736 L
F
.06221 .94986 m
.06329 .94986 L
.06329 .9736 L
.0627 .9736 L
F
.10127 .75989 m
.11392 .75989 L
.11392 .78363 L
.10127 .78363 L
F
.3038 .56991 m
.31646 .56991 L
.31646 .59366 L
.3038 .59366 L
F
.34177 .56991 m
.35443 .56991 L
.35443 .59366 L
.34177 .59366 L
F
.08861 .75989 m
.10127 .75989 L
.10127 .78363 L
.08861 .78363 L
F
.07595 .94986 m
.08861 .94986 L
.08861 .9736 L
.07595 .9736 L
F
.10127 .94986 m
.11392 .94986 L
.11392 .9736 L
.10127 .9736 L
F
.08576 .75989 m
.08861 .75989 L
.08861 .78363 L
.07841 .78363 L
F
.32911 1.3298 m
.34177 1.3298 L
.34177 1.35355 L
.32911 1.35355 L
F
.34177 1.3298 m
.35443 1.3298 L
.35443 1.35355 L
.34177 1.35355 L
F
.29114 1.3298 m
.3038 1.3298 L
.3038 1.35355 L
.29114 1.35355 L
F
.3038 1.3298 m
.31646 1.3298 L
.31646 1.35355 L
.3038 1.35355 L
F
.43038 .64115 m
.44304 .64115 L
.44304 .6649 L
.43038 .6649 L
F
.41772 .64115 m
.43038 .64115 L
.43038 .6649 L
.41772 .6649 L
F
.40506 .64115 m
.41772 .64115 L
.41772 .6649 L
.40506 .6649 L
F
.12658 .94986 m
.13924 .94986 L
.13924 .9736 L
.12658 .9736 L
F
.11392 .94986 m
.12658 .94986 L
.12658 .9736 L
.11392 .9736 L
F
.20253 .94986 m
.21519 .94986 L
.21519 .9736 L
.20253 .9736 L
F
.18987 .94986 m
.20253 .94986 L
.20253 .9736 L
.18987 .9736 L
F
.13924 .94986 m
.1519 .94986 L
.1519 .9736 L
.13924 .9736 L
F
.31646 1.3298 m
.32911 1.3298 L
.32911 1.35355 L
.31646 1.35355 L
F
.1519 .94986 m
.16456 .94986 L
.16456 .9736 L
.1519 .9736 L
F
.49367 1.3298 m
.50633 1.3298 L
.50633 1.35355 L
.49367 1.35355 L
F
.5443 1.3298 m
.55696 1.3298 L
.55696 1.35355 L
.5443 1.35355 L
F
.50633 1.3298 m
.51899 1.3298 L
.51899 1.35355 L
.50633 1.35355 L
F
.48101 1.3298 m
.49367 1.3298 L
.49367 1.35355 L
.48101 1.35355 L
F
.53165 1.3298 m
.5443 1.3298 L
.5443 1.35355 L
.53165 1.35355 L
F
.41139 1.3298 m
.41772 1.3298 L
.41772 1.34167 L
.41139 1.34167 L
F
.37975 .64115 m
.39241 .64115 L
.39241 .6649 L
.37975 .6649 L
F
.51899 1.3298 m
.53165 1.3298 L
.53165 1.35355 L
.51899 1.35355 L
F
.26582 .92611 m
.27848 .92611 L
.27848 .94986 L
.26582 .94986 L
F
.20253 .92611 m
.21519 .92611 L
.21519 .94986 L
.20253 .94986 L
F
.24051 .92611 m
.25316 .92611 L
.25316 .94986 L
.24051 .94986 L
F
.27848 .92611 m
.29114 .92611 L
.29114 .94986 L
.27848 .94986 L
F
.22785 .92611 m
.24051 .92611 L
.24051 .94986 L
.22785 .94986 L
F
.21519 .92611 m
.22785 .92611 L
.22785 .94986 L
.21519 .94986 L
F
.18987 .92611 m
.20253 .92611 L
.20253 .94986 L
.18987 .94986 L
F
.46835 .92611 m
.48101 .92611 L
.48101 .94986 L
.46835 .94986 L
F
.50633 .92611 m
.51899 .92611 L
.51899 .94986 L
.50633 .94986 L
F
.53165 .92611 m
.5443 .92611 L
.5443 .94986 L
.53165 .94986 L
F
.48101 .92611 m
.49367 .92611 L
.49367 .94986 L
.48101 .94986 L
F
.34177 .64115 m
.35443 .64115 L
.35443 .6649 L
.34177 .6649 L
F
.32911 .64115 m
.34177 .64115 L
.34177 .6649 L
.32911 .6649 L
F
.49367 .92611 m
.50633 .92611 L
.50633 .94986 L
.49367 .94986 L
F
.4557 1.3298 m
.46835 1.3298 L
.46835 1.35355 L
.4557 1.35355 L
F
.46835 1.3298 m
.48101 1.3298 L
.48101 1.35355 L
.46835 1.35355 L
F
.38924 1.3298 m
.39241 1.3298 L
.39241 1.33574 L
.38924 1.33574 L
F
.4019 1.3298 m
.40506 1.3298 L
.40506 1.33574 L
.4019 1.33574 L
F
.44304 1.3298 m
.4557 1.3298 L
.4557 1.35355 L
.44304 1.35355 L
F
.51899 .92611 m
.53165 .92611 L
.53165 .94986 L
.51899 .94986 L
F
.43038 1.3298 m
.44304 1.3298 L
.44304 1.35355 L
.43038 1.35355 L
F
.3038 .92611 m
.31646 .92611 L
.31646 .94986 L
.3038 .94986 L
F
.29114 .92611 m
.3038 .92611 L
.3038 .94986 L
.29114 .94986 L
F
.36709 .92611 m
.37975 .92611 L
.37975 .94986 L
.36709 .94986 L
F
.35443 .92611 m
.36709 .92611 L
.36709 .94986 L
.35443 .94986 L
F
.31646 .92611 m
.32911 .92611 L
.32911 .94986 L
.31646 .94986 L
F
.43038 .92611 m
.44304 .92611 L
.44304 .94986 L
.43038 .94986 L
F
.32911 .92611 m
.34177 .92611 L
.34177 .94986 L
.32911 .94986 L
F
.10127 1.0211 m
.11392 1.0211 L
.11392 1.04484 L
.10127 1.04484 L
F
.08861 1.0211 m
.10127 1.0211 L
.10127 1.04484 L
.08861 1.04484 L
F
.17722 1.0211 m
.18987 1.0211 L
.18987 1.04484 L
.17722 1.04484 L
F
.16456 1.0211 m
.17722 1.0211 L
.17722 1.04484 L
.16456 1.04484 L
F
.11392 1.0211 m
.12658 1.0211 L
.12658 1.04484 L
.11392 1.04484 L
F
.34177 .92611 m
.35443 .92611 L
.35443 .94986 L
.34177 .94986 L
F
.12658 1.0211 m
.13924 1.0211 L
.13924 1.04484 L
.12658 1.04484 L
F
.11392 .92611 m
.12658 .92611 L
.12658 .94986 L
.11392 .94986 L
F
.10127 .92611 m
.11392 .92611 L
.11392 .94986 L
.10127 .94986 L
F
.17722 .92611 m
.18987 .92611 L
.18987 .94986 L
.17722 .94986 L
F
.16456 .92611 m
.17722 .92611 L
.17722 .94986 L
.16456 .94986 L
F
.12658 .92611 m
.13924 .92611 L
.13924 .94986 L
.12658 .94986 L
F
.25316 .92611 m
.26582 .92611 L
.26582 .94986 L
.25316 .94986 L
F
.13924 .92611 m
.1519 .92611 L
.1519 .94986 L
.13924 .94986 L
F
.39241 .92611 m
.40506 .92611 L
.40506 .94986 L
.39241 .94986 L
F
.37975 .92611 m
.39241 .92611 L
.39241 .94986 L
.37975 .94986 L
F
.4557 .92611 m
.46835 .92611 L
.46835 .94986 L
.4557 .94986 L
F
.44304 .92611 m
.4557 .92611 L
.4557 .94986 L
.44304 .94986 L
F
.40506 .92611 m
.41772 .92611 L
.41772 .94986 L
.40506 .94986 L
F
.1519 .92611 m
.16456 .92611 L
.16456 .94986 L
.1519 .94986 L
F
.41772 .92611 m
.43038 .92611 L
.43038 .94986 L
.41772 .94986 L
F
.25316 .64115 m
.26582 .64115 L
.26582 .6649 L
.25316 .6649 L
F
.20253 .64115 m
.21519 .64115 L
.21519 .6649 L
.20253 .6649 L
F
.24051 .64115 m
.25316 .64115 L
.25316 .6649 L
.24051 .6649 L
F
.26582 .64115 m
.27848 .64115 L
.27848 .6649 L
.26582 .6649 L
F
.21519 .64115 m
.22785 .64115 L
.22785 .6649 L
.21519 .6649 L
F
.06329 .92611 m
.07595 .92611 L
.07595 .94986 L
.06329 .94986 L
F
.18987 .64115 m
.20253 .64115 L
.20253 .6649 L
.18987 .6649 L
F
.22785 .64115 m
.24051 .64115 L
.24051 .6649 L
.22785 .6649 L
F
.51899 .90236 m
.53165 .90236 L
.53165 .92611 L
.51899 .92611 L
F
.4557 .90236 m
.46835 .90236 L
.46835 .92611 L
.4557 .92611 L
F
.49367 .90236 m
.50633 .90236 L
.50633 .92611 L
.49367 .92611 L
F
.53165 .90236 m
.5443 .90236 L
.5443 .92611 L
.53165 .92611 L
F
.48101 .90236 m
.49367 .90236 L
.49367 .92611 L
.48101 .92611 L
F
.46835 .90236 m
.48101 .90236 L
.48101 .92611 L
.46835 .92611 L
F
.44304 .90236 m
.4557 .90236 L
.4557 .92611 L
.44304 .92611 L
F
.41772 .75989 m
.43038 .75989 L
.43038 .78363 L
.41772 .78363 L
F
.46835 .75989 m
.48101 .75989 L
.48101 .78363 L
.46835 .78363 L
F
.43038 .75989 m
.44304 .75989 L
.44304 .78363 L
.43038 .78363 L
F
.40506 .75989 m
.41772 .75989 L
.41772 .78363 L
.40506 .78363 L
F
.44304 .75989 m
.4557 .75989 L
.4557 .78363 L
.44304 .78363 L
F
.4557 .75989 m
.46835 .75989 L
.46835 .78363 L
.4557 .78363 L
F
.48101 .75989 m
.49367 .75989 L
.49367 .78363 L
.48101 .78363 L
F
.07595 .92611 m
.08861 .92611 L
.08861 .94986 L
.07595 .94986 L
F
.29114 .64115 m
.3038 .64115 L
.3038 .6649 L
.29114 .6649 L
F
.06221 .92611 m
.06329 .92611 L
.06329 .94986 L
.06221 .94986 L
F
.08861 .92611 m
.10127 .92611 L
.10127 .94986 L
.08861 .94986 L
F
.31646 .64115 m
.32911 .64115 L
.32911 .6649 L
.31646 .6649 L
F
.3038 .64115 m
.31646 .64115 L
.31646 .6649 L
.3038 .6649 L
F
.27848 .64115 m
.29114 .64115 L
.29114 .6649 L
.27848 .6649 L
F
.55696 .90236 m
.56962 .90236 L
.56962 .92611 L
.55696 .92611 L
F
.5443 .90236 m
.55696 .90236 L
.55696 .92611 L
.5443 .92611 L
F
.58228 .75989 m
.59494 .75989 L
.59494 .78363 L
.58228 .78363 L
F
.29114 .56991 m
.3038 .56991 L
.3038 .59366 L
.29114 .59366 L
F
.56962 .90236 m
.58228 .90236 L
.58228 .92611 L
.56962 .92611 L
F
.51899 .75989 m
.53165 .75989 L
.53165 .78363 L
.51899 .78363 L
F
.58228 .90236 m
.59494 .90236 L
.59494 .92611 L
.58228 .92611 L
F
.55696 .92611 m
.56962 .92611 L
.56962 .94986 L
.55696 .94986 L
F
.5443 .92611 m
.55696 .92611 L
.55696 .94986 L
.5443 .94986 L
F
.36709 .64115 m
.37975 .64115 L
.37975 .6649 L
.36709 .6649 L
F
.35443 .64115 m
.36709 .64115 L
.36709 .6649 L
.35443 .6649 L
F
.56962 .92611 m
.58228 .92611 L
.58228 .94986 L
.56962 .94986 L
F
.27848 .56991 m
.29114 .56991 L
.29114 .59366 L
.27848 .59366 L
F
.58228 .92611 m
.59494 .92611 L
.59494 .94986 L
.58228 .94986 L
F
.36709 .90236 m
.37975 .90236 L
.37975 .92611 L
.36709 .92611 L
F
.35443 .90236 m
.36709 .90236 L
.36709 .92611 L
.35443 .92611 L
F
.43038 .90236 m
.44304 .90236 L
.44304 .92611 L
.43038 .92611 L
F
.41772 .90236 m
.43038 .90236 L
.43038 .92611 L
.41772 .92611 L
F
.37975 .90236 m
.39241 .90236 L
.39241 .92611 L
.37975 .92611 L
F
.50633 .90236 m
.51899 .90236 L
.51899 .92611 L
.50633 .92611 L
F
.39241 .90236 m
.40506 .90236 L
.40506 .92611 L
.39241 .92611 L
F
.55696 .75989 m
.56962 .75989 L
.56962 .78363 L
.55696 .78363 L
F
.56962 .75989 m
.58228 .75989 L
.58228 .78363 L
.56962 .78363 L
F
.49367 .75989 m
.50633 .75989 L
.50633 .78363 L
.49367 .78363 L
F
.50633 .75989 m
.51899 .75989 L
.51899 .78363 L
.50633 .78363 L
F
.5443 .75989 m
.55696 .75989 L
.55696 .78363 L
.5443 .78363 L
F
.40506 .90236 m
.41772 .90236 L
.41772 .92611 L
.40506 .92611 L
F
.53165 .75989 m
.5443 .75989 L
.5443 .78363 L
.53165 .78363 L
F
.39241 .45118 m
.40506 .45118 L
.40506 .47493 L
.39241 .47493 L
F
.40506 .45118 m
.41772 .45118 L
.41772 .47493 L
.40506 .47493 L
F
.41772 .45118 m
.43038 .45118 L
.43038 .47493 L
.41772 .47493 L
F
.43038 .45118 m
.44304 .45118 L
.44304 .47493 L
.43038 .47493 L
F
.18987 .87862 m
.20253 .87862 L
.20253 .90236 L
.18987 .90236 L
F
.34177 .45118 m
.35443 .45118 L
.35443 .47493 L
.34177 .47493 L
F
.17722 .87862 m
.18987 .87862 L
.18987 .90236 L
.17722 .90236 L
F
.20253 .87862 m
.21519 .87862 L
.21519 .90236 L
.20253 .90236 L
F
.34177 .87862 m
.35443 .87862 L
.35443 .90236 L
.34177 .90236 L
F
.32911 .87862 m
.34177 .87862 L
.34177 .90236 L
.32911 .90236 L
F
.31646 .87862 m
.32911 .87862 L
.32911 .90236 L
.31646 .90236 L
F
.3038 .87862 m
.31646 .87862 L
.31646 .90236 L
.3038 .90236 L
F
.35443 .45118 m
.36709 .45118 L
.36709 .47493 L
.35443 .47493 L
F
.36709 .45118 m
.37975 .45118 L
.37975 .47493 L
.36709 .47493 L
F
.37975 .45118 m
.39241 .45118 L
.39241 .47493 L
.37975 .47493 L
F
.35443 1.35355 m
.36709 1.35355 L
.36709 1.37729 L
.35443 1.37729 L
F
.43038 1.35355 m
.44304 1.35355 L
.44304 1.37729 L
.43038 1.37729 L
F
.49367 1.35355 m
.50633 1.35355 L
.50633 1.37729 L
.49367 1.37729 L
F
.50633 1.35355 m
.51899 1.35355 L
.51899 1.37729 L
.50633 1.37729 L
F
.41772 1.35355 m
.43038 1.35355 L
.43038 1.37729 L
.41772 1.37729 L
F
.44304 .87862 m
.4557 .87862 L
.4557 .90236 L
.44304 .90236 L
F
.44304 1.35355 m
.4557 1.35355 L
.4557 1.37729 L
.44304 1.37729 L
F
.48101 1.35355 m
.49367 1.35355 L
.49367 1.37729 L
.48101 1.37729 L
F
.4557 1.35355 m
.46835 1.35355 L
.46835 1.37729 L
.4557 1.37729 L
F
.31646 .46457 m
.31646 .47493 L
.3038 .47493 L
.3038 .47264 L
F
.32911 .45709 m
.32911 .47493 L
.31646 .47493 L
.31646 .46457 L
F
.51899 1.35355 m
.53165 1.35355 L
.53165 1.37729 L
.51899 1.37729 L
F
.58228 .87862 m
.59494 .87862 L
.59494 .90236 L
.58228 .90236 L
F
.46835 1.35355 m
.48101 1.35355 L
.48101 1.37729 L
.46835 1.37729 L
F
.22785 .56991 m
.24051 .56991 L
.24051 .59366 L
.22785 .59366 L
F
.12658 .90236 m
.13924 .90236 L
.13924 .92611 L
.12658 .92611 L
F
.21519 .56991 m
.22785 .56991 L
.22785 .59366 L
.21519 .59366 L
F
.24051 .56991 m
.25316 .56991 L
.25316 .59366 L
.24051 .59366 L
F
.1519 .90236 m
.16456 .90236 L
.16456 .92611 L
.1519 .92611 L
F
.13924 .90236 m
.1519 .90236 L
.1519 .92611 L
.13924 .92611 L
F
.11392 .90236 m
.12658 .90236 L
.12658 .92611 L
.11392 .92611 L
F
.29114 .90236 m
.3038 .90236 L
.3038 .92611 L
.29114 .92611 L
F
.27848 .90236 m
.29114 .90236 L
.29114 .92611 L
.27848 .92611 L
F
.26582 .90236 m
.27848 .90236 L
.27848 .92611 L
.26582 .92611 L
F
.25316 .90236 m
.26582 .90236 L
.26582 .92611 L
.25316 .92611 L
F
.0627 .90236 m
.06329 .90236 L
.06329 .92611 L
.06221 .92611 L
F
.26582 .56991 m
.27848 .56991 L
.27848 .59366 L
.26582 .59366 L
F
.25316 .56991 m
.26582 .56991 L
.26582 .59366 L
.25316 .59366 L
F
.25316 .87862 m
.26582 .87862 L
.26582 .90236 L
.25316 .90236 L
F
.24051 .87862 m
.25316 .87862 L
.25316 .90236 L
.24051 .90236 L
F
.22785 .87862 m
.24051 .87862 L
.24051 .90236 L
.22785 .90236 L
F
.21519 .87862 m
.22785 .87862 L
.22785 .90236 L
.21519 .90236 L
F
.37975 .87862 m
.39241 .87862 L
.39241 .90236 L
.37975 .90236 L
F
.36709 .87862 m
.37975 .87862 L
.37975 .90236 L
.36709 .90236 L
F
.35443 .87862 m
.36709 .87862 L
.36709 .90236 L
.35443 .90236 L
F
.10127 .90236 m
.11392 .90236 L
.11392 .92611 L
.10127 .92611 L
F
.08861 .90236 m
.10127 .90236 L
.10127 .92611 L
.08861 .92611 L
F
.07595 .90236 m
.08861 .90236 L
.08861 .92611 L
.07595 .92611 L
F
.06329 .90236 m
.07595 .90236 L
.07595 .92611 L
.06329 .92611 L
F
.29114 .87862 m
.3038 .87862 L
.3038 .90236 L
.29114 .90236 L
F
.27848 .87862 m
.29114 .87862 L
.29114 .90236 L
.27848 .90236 L
F
.26582 .87862 m
.27848 .87862 L
.27848 .90236 L
.26582 .90236 L
F
.75633 1.51977 m
.75949 1.51977 L
.75949 1.52571 L
.75633 1.52571 L
F
.55696 .59366 m
.56962 .59366 L
.56962 .61741 L
.55696 .61741 L
F
.40506 .78363 m
.41772 .78363 L
.41772 .80738 L
.40506 .80738 L
F
.76899 1.51977 m
.77215 1.51977 L
.77215 1.52571 L
.76899 1.52571 L
F
.53165 .52242 m
.5443 .52242 L
.5443 .54617 L
.53165 .54617 L
F
.5443 .52242 m
.55696 .52242 L
.55696 .54617 L
.5443 .54617 L
F
.08861 .83112 m
.10127 .83112 L
.10127 .85487 L
.08861 .85487 L
F
.5443 .59366 m
.55696 .59366 L
.55696 .61741 L
.5443 .61741 L
F
.53165 .59366 m
.5443 .59366 L
.5443 .61741 L
.53165 .61741 L
F
.58228 .80738 m
.59494 .80738 L
.59494 .83112 L
.58228 .83112 L
F
.56962 .80738 m
.58228 .80738 L
.58228 .83112 L
.56962 .83112 L
F
.56962 .59366 m
.58228 .59366 L
.58228 .61741 L
.56962 .61741 L
F
.39241 .78363 m
.40506 .78363 L
.40506 .80738 L
.39241 .80738 L
F
.41772 .78363 m
.43038 .78363 L
.43038 .80738 L
.41772 .80738 L
F
.48101 .52242 m
.49367 .52242 L
.49367 .54617 L
.48101 .54617 L
F
.49367 .52242 m
.50633 .52242 L
.50633 .54617 L
.49367 .54617 L
F
.50633 .52242 m
.51899 .52242 L
.51899 .54617 L
.50633 .54617 L
F
.51899 .52242 m
.53165 .52242 L
.53165 .54617 L
.51899 .54617 L
F
.12658 .78363 m
.13924 .78363 L
.13924 .80738 L
.12658 .80738 L
F
.13924 .78363 m
.1519 .78363 L
.1519 .80738 L
.13924 .80738 L
F
.1519 .78363 m
.16456 .78363 L
.16456 .80738 L
.1519 .80738 L
F
.07595 .83112 m
.08861 .83112 L
.08861 .85487 L
.07595 .85487 L
F
.4557 .52242 m
.46835 .52242 L
.46835 .54617 L
.4557 .54617 L
F
.06876 .83112 m
.07595 .83112 L
.07595 .85487 L
.06581 .85487 L
F
.4557 1.37729 m
.46835 1.37729 L
.46835 1.40104 L
.4557 1.40104 L
F
.43038 .52242 m
.44304 .52242 L
.44304 .54617 L
.43038 .54617 L
F
.44304 .52242 m
.4557 .52242 L
.4557 .54617 L
.44304 .54617 L
F
.46835 .52242 m
.48101 .52242 L
.48101 .54617 L
.46835 .54617 L
F
.55696 .87862 m
.56962 .87862 L
.56962 .90236 L
.55696 .90236 L
F
.49367 .87862 m
.50633 .87862 L
.50633 .90236 L
.49367 .90236 L
F
.53165 .87862 m
.5443 .87862 L
.5443 .90236 L
.53165 .90236 L
F
.56962 .87862 m
.58228 .87862 L
.58228 .90236 L
.56962 .90236 L
F
.51899 .87862 m
.53165 .87862 L
.53165 .90236 L
.51899 .90236 L
F
.50633 .87862 m
.51899 .87862 L
.51899 .90236 L
.50633 .90236 L
F
.48101 .87862 m
.49367 .87862 L
.49367 .90236 L
.48101 .90236 L
F
.40506 .87862 m
.41772 .87862 L
.41772 .90236 L
.40506 .90236 L
F
.39241 .87862 m
.40506 .87862 L
.40506 .90236 L
.39241 .90236 L
F
.46835 .87862 m
.48101 .87862 L
.48101 .90236 L
.46835 .90236 L
F
.4557 .87862 m
.46835 .87862 L
.46835 .90236 L
.4557 .90236 L
F
.41772 .87862 m
.43038 .87862 L
.43038 .90236 L
.41772 .90236 L
F
.5443 .87862 m
.55696 .87862 L
.55696 .90236 L
.5443 .90236 L
F
.43038 .87862 m
.44304 .87862 L
.44304 .90236 L
.43038 .90236 L
F
.29114 .78363 m
.3038 .78363 L
.3038 .80738 L
.29114 .80738 L
F
.25316 .78363 m
.26582 .78363 L
.26582 .80738 L
.25316 .80738 L
F
.23418 .78363 m
.24051 .78363 L
.24051 .7955 L
.23418 .7955 L
F
.27848 .78363 m
.29114 .78363 L
.29114 .80738 L
.27848 .80738 L
F
.71984 1.51977 m
.72152 1.51977 L
.72152 1.53164 L
.71976 1.53164 L
F
.74367 1.51977 m
.74684 1.51977 L
.74684 1.52571 L
.74367 1.52571 L
F
.26582 .78363 m
.27848 .78363 L
.27848 .80738 L
.26582 .80738 L
F
.31646 .78363 m
.32911 .78363 L
.32911 .80738 L
.31646 .80738 L
F
.36709 .78363 m
.37975 .78363 L
.37975 .80738 L
.36709 .80738 L
F
.32911 .78363 m
.34177 .78363 L
.34177 .80738 L
.32911 .80738 L
F
.3038 .78363 m
.31646 .78363 L
.31646 .80738 L
.3038 .80738 L
F
.34177 .78363 m
.35443 .78363 L
.35443 .80738 L
.34177 .80738 L
F
.35443 .78363 m
.36709 .78363 L
.36709 .80738 L
.35443 .80738 L
F
.37975 .78363 m
.39241 .78363 L
.39241 .80738 L
.37975 .80738 L
F
.08861 .85487 m
.10127 .85487 L
.10127 .87862 L
.08861 .87862 L
F
.11392 .85487 m
.12658 .85487 L
.12658 .87862 L
.11392 .87862 L
F
.35443 .85487 m
.36709 .85487 L
.36709 .87862 L
.35443 .87862 L
F
.34177 .85487 m
.35443 .85487 L
.35443 .87862 L
.34177 .87862 L
F
.21519 .61741 m
.22785 .61741 L
.22785 .64115 L
.21519 .64115 L
F
.07595 .85487 m
.08861 .85487 L
.08861 .87862 L
.07595 .87862 L
F
.10127 .85487 m
.11392 .85487 L
.11392 .87862 L
.10127 .87862 L
F
.22785 .61741 m
.24051 .61741 L
.24051 .64115 L
.22785 .64115 L
F
.36709 .85487 m
.37975 .85487 L
.37975 .87862 L
.36709 .87862 L
F
.39241 .85487 m
.40506 .85487 L
.40506 .87862 L
.39241 .87862 L
F
.26582 .85487 m
.27848 .85487 L
.27848 .87862 L
.26582 .87862 L
F
.25316 .85487 m
.26582 .85487 L
.26582 .87862 L
.25316 .87862 L
F
.32911 .85487 m
.34177 .85487 L
.34177 .87862 L
.32911 .87862 L
F
.31646 .85487 m
.32911 .85487 L
.32911 .87862 L
.31646 .87862 L
F
.37975 .85487 m
.39241 .85487 L
.39241 .87862 L
.37975 .87862 L
F
.31646 .47493 m
.32911 .47493 L
.32911 .49867 L
.31646 .49867 L
F
.3038 .47493 m
.31646 .47493 L
.31646 .49867 L
.3038 .49867 L
F
.16456 .85487 m
.17722 .85487 L
.17722 .87862 L
.16456 .87862 L
F
.1519 .85487 m
.16456 .85487 L
.16456 .87862 L
.1519 .87862 L
F
.36709 .47493 m
.37975 .47493 L
.37975 .49867 L
.36709 .49867 L
F
.32911 .47493 m
.34177 .47493 L
.34177 .49867 L
.32911 .49867 L
F
.35443 .47493 m
.36709 .47493 L
.36709 .49867 L
.35443 .49867 L
F
.17722 .85487 m
.18987 .85487 L
.18987 .87862 L
.17722 .87862 L
F
.21519 .85487 m
.22785 .85487 L
.22785 .87862 L
.21519 .87862 L
F
.06581 .85487 m
.07595 .85487 L
.07595 .87862 L
.06382 .87862 L
F
.18987 .85487 m
.20253 .85487 L
.20253 .87862 L
.18987 .87862 L
F
.13924 .85487 m
.1519 .85487 L
.1519 .87862 L
.13924 .87862 L
F
.12658 .85487 m
.13924 .85487 L
.13924 .87862 L
.12658 .87862 L
F
.20253 .85487 m
.21519 .85487 L
.21519 .87862 L
.20253 .87862 L
F
.51899 .4712 m
.51899 .47493 L
.50633 .47493 L
.50633 .46364 L
F
.56962 .61741 m
.58228 .61741 L
.58228 .64115 L
.56962 .64115 L
F
.1519 .87862 m
.16456 .87862 L
.16456 .90236 L
.1519 .90236 L
F
.50633 .46364 m
.50633 .47493 L
.49367 .47493 L
.49367 .45678 L
F
.53165 .61741 m
.5443 .61741 L
.5443 .64115 L
.53165 .64115 L
F
.51899 .61741 m
.53165 .61741 L
.53165 .64115 L
.51899 .64115 L
F
.07595 .80738 m
.08861 .80738 L
.08861 .83112 L
.07595 .83112 L
F
.08861 .87862 m
.10127 .87862 L
.10127 .90236 L
.08861 .90236 L
F
.46835 .45118 m
.48101 .45118 L
.48101 .47493 L
.46835 .47493 L
F
.07595 .87862 m
.08861 .87862 L
.08861 .90236 L
.07595 .90236 L
F
.10127 .87862 m
.11392 .87862 L
.11392 .90236 L
.10127 .90236 L
F
.44304 .45118 m
.4557 .45118 L
.4557 .47493 L
.44304 .47493 L
F
.4557 .45118 m
.46835 .45118 L
.46835 .47493 L
.4557 .47493 L
F
.16456 .87862 m
.17722 .87862 L
.17722 .90236 L
.16456 .90236 L
F
.27848 .85487 m
.29114 .85487 L
.29114 .87862 L
.27848 .87862 L
F
.3038 .85487 m
.31646 .85487 L
.31646 .87862 L
.3038 .87862 L
F
.49367 .59366 m
.50633 .59366 L
.50633 .61741 L
.49367 .61741 L
F
.50633 .59366 m
.51899 .59366 L
.51899 .61741 L
.50633 .61741 L
F
.24051 .85487 m
.25316 .85487 L
.25316 .87862 L
.24051 .87862 L
F
.22785 .85487 m
.24051 .85487 L
.24051 .87862 L
.22785 .87862 L
F
.29114 .85487 m
.3038 .85487 L
.3038 .87862 L
.29114 .87862 L
F
.46835 .59366 m
.48101 .59366 L
.48101 .61741 L
.46835 .61741 L
F
.48101 .59366 m
.49367 .59366 L
.49367 .61741 L
.48101 .61741 L
F
.55696 .61741 m
.56962 .61741 L
.56962 .64115 L
.55696 .64115 L
F
.5443 .61741 m
.55696 .61741 L
.55696 .64115 L
.5443 .64115 L
F
.51899 .59366 m
.53165 .59366 L
.53165 .61741 L
.51899 .61741 L
F
.07284 .80738 m
.07595 .80738 L
.07595 .83112 L
.06876 .83112 L
F
.53165 1.35355 m
.5443 1.35355 L
.5443 1.37729 L
.53165 1.37729 L
F
.34177 1.35355 m
.35443 1.35355 L
.35443 1.37729 L
.34177 1.37729 L
F
.3038 1.35355 m
.31646 1.35355 L
.31646 1.37729 L
.3038 1.37729 L
F
.27848 1.35355 m
.29114 1.35355 L
.29114 1.37729 L
.27848 1.37729 L
F
.32911 1.35355 m
.34177 1.35355 L
.34177 1.37729 L
.32911 1.37729 L
F
.43038 .59366 m
.44304 .59366 L
.44304 .61741 L
.43038 .61741 L
F
.44304 .59366 m
.4557 .59366 L
.4557 .61741 L
.44304 .61741 L
F
.31646 1.35355 m
.32911 1.35355 L
.32911 1.37729 L
.31646 1.37729 L
F
.41772 .43737 m
.41772 .45118 L
.40506 .45118 L
.40506 .43724 L
F
.37975 .43982 m
.37975 .45118 L
.36709 .45118 L
.36709 .44252 L
F
.35443 .44614 m
.35443 .45118 L
.34177 .45118 L
.34177 .45061 L
F
.40506 .43724 m
.40506 .45118 L
.39241 .45118 L
.39241 .43806 L
F
.29114 1.35355 m
.3038 1.35355 L
.3038 1.37729 L
.29114 1.37729 L
F
.4557 .59366 m
.46835 .59366 L
.46835 .61741 L
.4557 .61741 L
F
.39241 .43806 m
.39241 .45118 L
.37975 .45118 L
.37975 .43982 L
F
.18987 .90236 m
.20253 .90236 L
.20253 .92611 L
.18987 .92611 L
F
.31646 .90236 m
.32911 .90236 L
.32911 .92611 L
.31646 .92611 L
F
.16456 .90236 m
.17722 .90236 L
.17722 .92611 L
.16456 .92611 L
F
.20253 .90236 m
.21519 .90236 L
.21519 .92611 L
.20253 .92611 L
F
.34177 .90236 m
.35443 .90236 L
.35443 .92611 L
.34177 .92611 L
F
.32911 .90236 m
.34177 .90236 L
.34177 .92611 L
.32911 .92611 L
F
.3038 .90236 m
.31646 .90236 L
.31646 .92611 L
.3038 .92611 L
F
.74684 1.49343 m
.74684 1.49602 L
.74051 1.49602 L
.73855 1.49602 L
F
.21519 .90236 m
.22785 .90236 L
.22785 .92611 L
.21519 .92611 L
F
.40506 .59366 m
.41772 .59366 L
.41772 .61741 L
.40506 .61741 L
F
.41772 .59366 m
.43038 .59366 L
.43038 .61741 L
.41772 .61741 L
F
.22785 .90236 m
.24051 .90236 L
.24051 .92611 L
.22785 .92611 L
F
.17722 .90236 m
.18987 .90236 L
.18987 .92611 L
.17722 .92611 L
F
.24051 .90236 m
.25316 .90236 L
.25316 .92611 L
.24051 .92611 L
F
.48101 .47493 m
.49367 .47493 L
.49367 .49867 L
.48101 .49867 L
F
.49367 .47493 m
.50633 .47493 L
.50633 .49867 L
.49367 .49867 L
F
.43038 .85487 m
.44304 .85487 L
.44304 .87862 L
.43038 .87862 L
F
.50633 .47493 m
.51899 .47493 L
.51899 .49867 L
.50633 .49867 L
F
.41772 .85487 m
.43038 .85487 L
.43038 .87862 L
.41772 .87862 L
F
.5443 .49236 m
.5443 .49867 L
.53165 .49867 L
.53165 .48079 L
F
.40506 .85487 m
.41772 .85487 L
.41772 .87862 L
.40506 .87862 L
F
.4557 .85487 m
.46835 .85487 L
.46835 .87862 L
.4557 .87862 L
F
.49367 .85487 m
.50633 .85487 L
.50633 .87862 L
.49367 .87862 L
F
.34177 .47493 m
.35443 .47493 L
.35443 .49867 L
.34177 .49867 L
F
.46835 .85487 m
.48101 .85487 L
.48101 .87862 L
.46835 .87862 L
F
.29114 .48305 m
.29114 .49867 L
.27848 .49867 L
.27848 .49417 L
F
.44304 .85487 m
.4557 .85487 L
.4557 .87862 L
.44304 .87862 L
F
.48101 .85487 m
.49367 .85487 L
.49367 .87862 L
.48101 .87862 L
F
.43038 .43841 m
.43038 .45118 L
.41772 .45118 L
.41772 .43737 L
F
.44304 .44032 m
.44304 .45118 L
.43038 .45118 L
.43038 .43841 L
F
.44304 .47493 m
.4557 .47493 L
.4557 .49867 L
.44304 .49867 L
F
.4557 .44306 m
.4557 .45118 L
.44304 .45118 L
.44304 .44032 L
F
.48101 .45077 m
.48101 .45118 L
.46835 .45118 L
.46835 .44656 L
F
.36709 .44252 m
.36709 .45118 L
.35443 .45118 L
.35443 .44614 L
F
.46835 .44656 m
.46835 .45118 L
.4557 .45118 L
.4557 .44306 L
F
.37975 .47493 m
.39241 .47493 L
.39241 .49867 L
.37975 .49867 L
F
.39241 .47493 m
.40506 .47493 L
.40506 .49867 L
.39241 .49867 L
F
.40506 .47493 m
.41772 .47493 L
.41772 .49867 L
.40506 .49867 L
F
.41772 .47493 m
.43038 .47493 L
.43038 .49867 L
.41772 .49867 L
F
.46835 .47493 m
.48101 .47493 L
.48101 .49867 L
.46835 .49867 L
F
.43038 .47493 m
.44304 .47493 L
.44304 .49867 L
.43038 .49867 L
F
.4557 .47493 m
.46835 .47493 L
.46835 .49867 L
.4557 .49867 L
F
.35443 1.37729 m
.36709 1.37729 L
.36709 1.40104 L
.35443 1.40104 L
F
.36709 1.37729 m
.37975 1.37729 L
.37975 1.40104 L
.36709 1.40104 L
F
.35443 .49867 m
.36709 .49867 L
.36709 .52242 L
.35443 .52242 L
F
.37975 1.37729 m
.39241 1.37729 L
.39241 1.40104 L
.37975 1.40104 L
F
.43038 1.37729 m
.44304 1.37729 L
.44304 1.40104 L
.43038 1.40104 L
F
.39241 1.37729 m
.40506 1.37729 L
.40506 1.40104 L
.39241 1.40104 L
F
.41772 1.37729 m
.43038 1.37729 L
.43038 1.40104 L
.41772 1.40104 L
F
.44304 .49867 m
.4557 .49867 L
.4557 .52242 L
.44304 .52242 L
F
.4557 .49867 m
.46835 .49867 L
.46835 .52242 L
.4557 .52242 L
F
.48101 .49867 m
.49367 .49867 L
.49367 .52242 L
.48101 .52242 L
F
.31646 .49867 m
.32911 .49867 L
.32911 .52242 L
.31646 .52242 L
F
.32911 .49867 m
.34177 .49867 L
.34177 .52242 L
.32911 .52242 L
F
.34177 .49867 m
.35443 .49867 L
.35443 .52242 L
.34177 .52242 L
F
.31646 1.37729 m
.32911 1.37729 L
.32911 1.40104 L
.31646 1.40104 L
F
.1519 .80738 m
.16456 .80738 L
.16456 .83112 L
.1519 .83112 L
F
.34177 1.37729 m
.35443 1.37729 L
.35443 1.40104 L
.34177 1.40104 L
F
.13924 .80738 m
.1519 .80738 L
.1519 .83112 L
.13924 .83112 L
F
.18987 .80738 m
.20253 .80738 L
.20253 .83112 L
.18987 .83112 L
F
.12658 .80738 m
.13924 .80738 L
.13924 .83112 L
.12658 .83112 L
F
.51899 .49867 m
.53165 .49867 L
.53165 .52242 L
.51899 .52242 L
F
.53165 .49867 m
.5443 .49867 L
.5443 .52242 L
.53165 .52242 L
F
.40506 1.37729 m
.41772 1.37729 L
.41772 1.40104 L
.40506 1.40104 L
F
.50633 .49867 m
.51899 .49867 L
.51899 .52242 L
.50633 .52242 L
F
.32911 1.37729 m
.34177 1.37729 L
.34177 1.40104 L
.32911 1.40104 L
F
.49367 .49867 m
.50633 .49867 L
.50633 .52242 L
.49367 .52242 L
F
.24051 .54617 m
.25316 .54617 L
.25316 .56991 L
.24051 .56991 L
F
.49367 .54617 m
.50633 .54617 L
.50633 .56991 L
.49367 .56991 L
F
.26582 .54617 m
.27848 .54617 L
.27848 .56991 L
.26582 .56991 L
F
.48101 .54617 m
.49367 .54617 L
.49367 .56991 L
.48101 .56991 L
F
.53165 .54617 m
.5443 .54617 L
.5443 .56991 L
.53165 .56991 L
F
.46835 .54617 m
.48101 .54617 L
.48101 .56991 L
.46835 .56991 L
F
.32911 .54617 m
.34177 .54617 L
.34177 .56991 L
.32911 .56991 L
F
.22785 .55076 m
.22785 .56991 L
.21519 .56991 L
.21519 .5662 L
F
.31646 .54617 m
.32911 .54617 L
.32911 .56991 L
.31646 .56991 L
F
.5443 .80738 m
.55696 .80738 L
.55696 .83112 L
.5443 .83112 L
F
.25316 .54617 m
.26582 .54617 L
.26582 .56991 L
.25316 .56991 L
F
.55696 .80738 m
.56962 .80738 L
.56962 .83112 L
.55696 .83112 L
F
.4557 .54617 m
.46835 .54617 L
.46835 .56991 L
.4557 .56991 L
F
.43038 .54617 m
.44304 .54617 L
.44304 .56991 L
.43038 .56991 L
F
.44304 .54617 m
.4557 .54617 L
.4557 .56991 L
.44304 .56991 L
F
.41772 .49867 m
.43038 .49867 L
.43038 .52242 L
.41772 .52242 L
F
.46835 .49867 m
.48101 .49867 L
.48101 .52242 L
.46835 .52242 L
F
.43038 .49867 m
.44304 .49867 L
.44304 .52242 L
.43038 .52242 L
F
.50633 .54617 m
.51899 .54617 L
.51899 .56991 L
.50633 .56991 L
F
.51899 .54617 m
.53165 .54617 L
.53165 .56991 L
.51899 .56991 L
F
.5443 .54617 m
.55696 .54617 L
.55696 .56991 L
.5443 .56991 L
F
.41139 .54617 m
.41772 .54617 L
.41772 .55804 L
.41139 .55804 L
F
.38924 .54617 m
.39241 .54617 L
.39241 .5521 L
.38924 .5521 L
F
.4019 .54617 m
.40506 .54617 L
.40506 .5521 L
.4019 .5521 L
F
.37342 .52242 m
.37975 .52242 L
.37975 .53429 L
.37342 .53429 L
F
.34177 .52242 m
.35443 .52242 L
.35443 .54617 L
.34177 .54617 L
F
.21519 .83112 m
.22785 .83112 L
.22785 .85487 L
.21519 .85487 L
F
.20253 .83112 m
.21519 .83112 L
.21519 .85487 L
.20253 .85487 L
F
.26582 .52242 m
.27848 .52242 L
.27848 .54617 L
.26582 .54617 L
F
.27848 .52242 m
.29114 .52242 L
.29114 .54617 L
.27848 .54617 L
F
.41139 .52242 m
.41772 .52242 L
.41772 .53429 L
.41139 .53429 L
F
.11392 .83112 m
.12658 .83112 L
.12658 .85487 L
.11392 .85487 L
F
.25316 .83112 m
.26582 .83112 L
.26582 .85487 L
.25316 .85487 L
F
.12658 .83112 m
.13924 .83112 L
.13924 .85487 L
.12658 .85487 L
F
.18987 .83112 m
.20253 .83112 L
.20253 .85487 L
.18987 .85487 L
F
.17722 .83112 m
.18987 .83112 L
.18987 .85487 L
.17722 .85487 L
F
.24051 .83112 m
.25316 .83112 L
.25316 .85487 L
.24051 .85487 L
F
.51899 .83112 m
.53165 .83112 L
.53165 .85487 L
.51899 .85487 L
F
.46835 .83112 m
.48101 .83112 L
.48101 .85487 L
.46835 .85487 L
F
.53165 .83112 m
.5443 .83112 L
.5443 .85487 L
.53165 .85487 L
F
.49367 .83112 m
.50633 .83112 L
.50633 .85487 L
.49367 .85487 L
F
.48101 .83112 m
.49367 .83112 L
.49367 .85487 L
.48101 .85487 L
F
.4557 .83112 m
.46835 .83112 L
.46835 .85487 L
.4557 .85487 L
F
.31646 .52242 m
.32911 .52242 L
.32911 .54617 L
.31646 .54617 L
F
.32911 .52242 m
.34177 .52242 L
.34177 .54617 L
.32911 .54617 L
F
.25316 .52242 m
.26582 .52242 L
.26582 .54617 L
.25316 .54617 L
F
.3038 .52242 m
.31646 .52242 L
.31646 .54617 L
.3038 .54617 L
F
.50633 .83112 m
.51899 .83112 L
.51899 .85487 L
.50633 .85487 L
F
.29114 .52242 m
.3038 .52242 L
.3038 .54617 L
.29114 .54617 L
F
.11392 .80738 m
.12658 .80738 L
.12658 .83112 L
.11392 .83112 L
F
.08861 .80738 m
.10127 .80738 L
.10127 .83112 L
.08861 .83112 L
F
.10127 .80738 m
.11392 .80738 L
.11392 .83112 L
.10127 .83112 L
F
.3038 1.37729 m
.31646 1.37729 L
.31646 1.40104 L
.3038 1.40104 L
F
.29114 .49867 m
.3038 .49867 L
.3038 .52242 L
.29114 .52242 L
F
.26582 .5071 m
.26582 .52242 L
.25316 .52242 L
.25316 .52048 L
F
.16456 .80738 m
.17722 .80738 L
.17722 .83112 L
.16456 .83112 L
F
.17722 .80738 m
.18987 .80738 L
.18987 .83112 L
.17722 .83112 L
F
.46835 1.40104 m
.48101 1.40104 L
.48101 1.42478 L
.46835 1.42478 L
F
.20253 .61741 m
.21519 .61741 L
.21519 .64115 L
.20253 .64115 L
F
.18987 .61741 m
.20253 .61741 L
.20253 .64115 L
.18987 .64115 L
F
.17722 .61741 m
.18987 .61741 L
.18987 .64115 L
.17722 .64115 L
F
.40506 1.40104 m
.41772 1.40104 L
.41772 1.42478 L
.40506 1.42478 L
F
.13924 .83112 m
.1519 .83112 L
.1519 .85487 L
.13924 .85487 L
F
.39241 1.40104 m
.40506 1.40104 L
.40506 1.42478 L
.39241 1.42478 L
F
.1519 .83112 m
.16456 .83112 L
.16456 .85487 L
.1519 .85487 L
F
.10127 .83112 m
.11392 .83112 L
.11392 .85487 L
.10127 .85487 L
F
.16456 .83112 m
.17722 .83112 L
.17722 .85487 L
.16456 .85487 L
F
.27848 .49867 m
.29114 .49867 L
.29114 .52242 L
.27848 .52242 L
F
.43038 1.40104 m
.44304 1.40104 L
.44304 1.42478 L
.43038 1.42478 L
F
.3038 .49867 m
.31646 .49867 L
.31646 .52242 L
.3038 .52242 L
F
.4557 1.40104 m
.46835 1.40104 L
.46835 1.42478 L
.4557 1.42478 L
F
.44304 1.40104 m
.4557 1.40104 L
.4557 1.42478 L
.44304 1.42478 L
F
.41772 1.40104 m
.43038 1.40104 L
.43038 1.42478 L
.41772 1.42478 L
F
.41772 .80738 m
.43038 .80738 L
.43038 .83112 L
.41772 .83112 L
F
.55696 .54617 m
.56962 .54617 L
.56962 .56991 L
.55696 .56991 L
F
.39241 .80738 m
.40506 .80738 L
.40506 .83112 L
.39241 .83112 L
F
.43038 .80738 m
.44304 .80738 L
.44304 .83112 L
.43038 .83112 L
F
.49367 .80738 m
.50633 .80738 L
.50633 .83112 L
.49367 .83112 L
F
.32911 .80738 m
.34177 .80738 L
.34177 .83112 L
.32911 .83112 L
F
.53165 .80738 m
.5443 .80738 L
.5443 .83112 L
.53165 .83112 L
F
.46835 .80738 m
.48101 .80738 L
.48101 .83112 L
.46835 .83112 L
F
.4557 .80738 m
.46835 .80738 L
.46835 .83112 L
.4557 .83112 L
F
.08861 .78363 m
.10127 .78363 L
.10127 .80738 L
.08861 .80738 L
F
.44304 .80738 m
.4557 .80738 L
.4557 .83112 L
.44304 .83112 L
F
.40506 .80738 m
.41772 .80738 L
.41772 .83112 L
.40506 .83112 L
F
.31646 .80738 m
.32911 .80738 L
.32911 .83112 L
.31646 .83112 L
F
.3038 .54617 m
.31646 .54617 L
.31646 .56991 L
.3038 .56991 L
F
.3038 .80738 m
.31646 .80738 L
.31646 .83112 L
.3038 .83112 L
F
.35443 .80738 m
.36709 .80738 L
.36709 .83112 L
.35443 .83112 L
F
.20253 .80738 m
.21519 .80738 L
.21519 .83112 L
.20253 .83112 L
F
.23418 .80738 m
.24051 .80738 L
.24051 .81925 L
.23418 .81925 L
F
.36709 .80738 m
.37975 .80738 L
.37975 .83112 L
.36709 .83112 L
F
.37975 .80738 m
.39241 .80738 L
.39241 .83112 L
.37975 .83112 L
F
.34177 .80738 m
.35443 .80738 L
.35443 .83112 L
.34177 .83112 L
F
.51899 .80738 m
.53165 .80738 L
.53165 .83112 L
.51899 .83112 L
F
.50633 .80738 m
.51899 .80738 L
.51899 .83112 L
.50633 .83112 L
F
.29114 .80738 m
.3038 .80738 L
.3038 .83112 L
.29114 .83112 L
F
.48101 .80738 m
.49367 .80738 L
.49367 .83112 L
.48101 .83112 L
F
.27848 .80738 m
.29114 .80738 L
.29114 .83112 L
.27848 .83112 L
F
.26582 .80738 m
.27848 .80738 L
.27848 .83112 L
.26582 .83112 L
F
.25316 .80738 m
.26582 .80738 L
.26582 .83112 L
.25316 .83112 L
F
.50633 1.37729 m
.51899 1.37729 L
.51899 1.40104 L
.50633 1.40104 L
F
.46835 1.37729 m
.48101 1.37729 L
.48101 1.40104 L
.46835 1.40104 L
F
.34177 .54617 m
.35443 .54617 L
.35443 .56991 L
.34177 .56991 L
F
.48101 1.37729 m
.49367 1.37729 L
.49367 1.40104 L
.48101 1.40104 L
F
.17722 .78363 m
.18987 .78363 L
.18987 .80738 L
.17722 .80738 L
F
.27848 .54617 m
.29114 .54617 L
.29114 .56991 L
.27848 .56991 L
F
.29114 .54617 m
.3038 .54617 L
.3038 .56991 L
.29114 .56991 L
F
.18987 .78363 m
.20253 .78363 L
.20253 .80738 L
.18987 .80738 L
F
.16456 .78363 m
.17722 .78363 L
.17722 .80738 L
.16456 .80738 L
F
.10127 .78363 m
.11392 .78363 L
.11392 .80738 L
.10127 .80738 L
F
.20253 .78363 m
.21519 .78363 L
.21519 .80738 L
.20253 .80738 L
F
.49367 1.37729 m
.50633 1.37729 L
.50633 1.40104 L
.49367 1.40104 L
F
.11392 .78363 m
.12658 .78363 L
.12658 .80738 L
.11392 .80738 L
F
.48213 .45118 m
.49367 .45678 L
.49367 .47493 L
.48101 .47493 L
.48101 .45118 L
F
.09509 1.13983 m
.08861 1.12365 L
.08861 1.11608 L
.10127 1.11608 L
.10127 1.13983 L
F
.24051 .80738 m
.25316 .80738 L
.25316 .83112 L
.24051 .83112 L
.24051 .81925 L
F
.21519 .80738 m
.22785 .80738 L
.22785 .81925 L
.22785 .83112 L
.21519 .83112 L
F
.34041 1.42478 m
.32911 1.41887 L
.32911 1.40104 L
.34177 1.40104 L
.34177 1.42478 L
F
.24051 .78363 m
.25316 .78363 L
.25316 .80738 L
.24051 .80738 L
.24051 .7955 L
F
.24051 1.06859 m
.25316 1.06859 L
.25316 1.09233 L
.24051 1.09233 L
.24051 1.08046 L
F
.57892 .54617 m
.58228 .55654 L
.58228 .56991 L
.56962 .56991 L
.56962 .54617 L
F
.21519 1.06859 m
.22785 1.06859 L
.22785 1.08046 L
.22785 1.09233 L
.21519 1.09233 L
F
.17285 .61741 m
.17722 .61741 L
.17722 .64115 L
.16456 .64115 L
.16456 .62773 L
F
.72785 .34432 m
.73418 .34432 L
.73418 .35026 L
.73418 .3562 L
.72785 .3562 L
F
.74684 .31377 m
.75316 .3125 L
.75316 .32058 L
.75 .32058 L
.74684 .32058 L
F
.74684 .31377 m
.74684 .32058 L
.74367 .32058 L
.74051 .32058 L
.74051 .31629 L
F
.41139 1.34167 m
.40506 1.34167 L
.40506 1.33574 L
.40506 1.3298 L
.41139 1.3298 L
F
.10649 .71239 m
.11392 .71239 L
.11392 .73614 L
.10127 .73614 L
.10127 .72292 L
F
.06382 .87862 m
.07595 .87862 L
.07595 .90236 L
.06329 .90236 L
.06329 .88914 L
F
.13595 1.21107 m
.12658 1.19722 L
.12658 1.18732 L
.13924 1.18732 L
.13924 1.21107 L
F
.75949 .31248 m
.76582 .31374 L
.76582 .32058 L
.76266 .32058 L
.75949 .32058 L
F
.09509 .73614 m
.10127 .73614 L
.10127 .75989 L
.08861 .75989 L
.08861 .75232 L
F
.13595 .6649 m
.13924 .6649 L
.13924 .68865 L
.12658 .68865 L
.12658 .67875 L
F
.77215 .31641 m
.77215 .32058 L
.76899 .32058 L
.76582 .32058 L
.76582 .31374 L
F
.07841 1.09233 m
.07595 1.08224 L
.07595 1.06859 L
.08861 1.06859 L
.08861 1.09233 L
F
.75949 .31248 m
.75949 .32058 L
.75633 .32058 L
.75316 .32058 L
.75316 .3125 L
F
.23418 1.09233 m
.24051 1.09233 L
.24051 1.11608 L
.22785 1.11608 L
.22785 1.09233 L
F
.49367 1.41919 m
.48213 1.42478 L
.48101 1.42478 L
.48101 1.40104 L
.49367 1.40104 L
F
.55078 .49867 m
.55696 .507 L
.55696 .52242 L
.5443 .52242 L
.5443 .49867 L
F
.52483 .47493 m
.53165 .48079 L
.53165 .49867 L
.51899 .49867 L
.51899 .47493 L
F
.37342 1.3298 m
.37975 1.3298 L
.37975 1.33574 L
.37975 1.34167 L
.37342 1.34167 L
F
.35443 1.3298 m
.36709 1.3298 L
.36709 1.34167 L
.36709 1.35355 L
.35443 1.35355 L
F
.23418 .83112 m
.24051 .83112 L
.24051 .85487 L
.22785 .85487 L
.22785 .83112 L
F
.39873 .52242 m
.40506 .52242 L
.40506 .53429 L
.4019 .53429 L
.39873 .53429 L
F
.53165 1.39518 m
.52483 1.40104 L
.51899 1.40104 L
.51899 1.37729 L
.53165 1.37729 L
F
.56962 1.34967 m
.56771 1.35355 L
.55696 1.35355 L
.55696 1.3298 L
.56962 1.3298 L
F
.74367 .3562 m
.74684 .3562 L
.74684 .36807 L
.74051 .36807 L
.74051 .3562 L
F
.30044 1.40104 m
.29114 1.39292 L
.29114 1.37729 L
.3038 1.37729 L
.3038 1.40104 L
F
.35443 .52242 m
.36709 .52242 L
.36709 .53429 L
.36709 .54617 L
.35443 .54617 L
F
.55696 1.36897 m
.55078 1.37729 L
.5443 1.37729 L
.5443 1.35355 L
.55696 1.35355 L
F
.07841 .78363 m
.08861 .78363 L
.08861 .80738 L
.07595 .80738 L
.07595 .79373 L
F
.12015 .68865 m
.12658 .68865 L
.12658 .71239 L
.11392 .71239 L
.11392 .69915 L
F
.59465 .61741 m
.59494 .62067 L
.59494 .64115 L
.58228 .64115 L
.58228 .61741 L
F
.38608 .52242 m
.39241 .52242 L
.39241 .53429 L
.38924 .53429 L
.38608 .53429 L
F
.41772 .52242 m
.43038 .52242 L
.43038 .54617 L
.41772 .54617 L
.41772 .53429 L
F
.77848 1.51977 m
.78481 1.51977 L
.78481 1.53164 L
.77848 1.53164 L
.77848 1.52571 L
F
.21519 1.04484 m
.22785 1.04484 L
.22785 1.05672 L
.22785 1.06859 L
.21519 1.06859 L
F
.56771 .52242 m
.56962 .52629 L
.56962 .54617 L
.55696 .54617 L
.55696 .52242 L
F
.22785 1.0211 m
.24051 1.0211 L
.24051 1.04484 L
.23418 1.04484 L
.22785 1.04484 L
F
.27338 .49867 m
.27848 .49867 L
.27848 .52242 L
.26582 .52242 L
.26582 .5071 L
F
.06382 .99735 m
.06329 .98683 L
.06329 .9736 L
.07595 .9736 L
.07595 .99735 L
F
.21519 .78363 m
.22785 .78363 L
.22785 .7955 L
.22785 .80738 L
.21519 .80738 L
F
.24051 1.04484 m
.25316 1.04484 L
.25316 1.06859 L
.24051 1.06859 L
.24051 1.05672 L
F
.30044 .47493 m
.3038 .47493 L
.3038 .49867 L
.29114 .49867 L
.29114 .48305 L
F
.25129 1.35355 m
.24051 1.34018 L
.24051 1.3298 L
.25316 1.3298 L
.25316 1.35355 L
F
.78481 .3626 m
.78131 .36807 L
.77848 .36807 L
.77848 .3562 L
.78481 .3562 L
F
.25129 .52242 m
.25316 .52242 L
.25316 .54617 L
.24051 .54617 L
.24051 .53579 L
F
.75949 .38202 m
.74684 .38254 L
.74684 .37994 L
.75316 .37994 L
.75949 .37994 L
F
.76899 .3562 m
.77215 .3562 L
.77215 .36807 L
.76582 .36807 L
.76582 .3562 L
F
.72785 1.51977 m
.73418 1.51977 L
.73418 1.52571 L
.73418 1.53164 L
.72785 1.53164 L
F
.75633 .3562 m
.75949 .3562 L
.75949 .36807 L
.75316 .36807 L
.75316 .3562 L
F
.27338 1.37729 m
.26582 1.36887 L
.26582 1.35355 L
.27848 1.35355 L
.27848 1.37729 L
F
.37342 1.35355 m
.37975 1.35355 L
.37975 1.37729 L
.36709 1.37729 L
.36709 1.35355 L
F
.37342 1.31793 m
.37975 1.31793 L
.37975 1.32386 L
.37975 1.3298 L
.37342 1.3298 L
F
.73855 .37994 m
.73418 .37773 L
.73418 .36807 L
.74051 .36807 L
.74051 .37994 L
F
.77848 1.53164 m
.78481 1.53164 L
.78481 1.54352 L
.77848 1.54352 L
.77848 1.53758 L
F
.76582 .37984 m
.76546 .37994 L
.75949 .37994 L
.75949 .36807 L
.76582 .36807 L
F
.38291 1.31793 m
.37975 1.31793 L
.37975 1.30605 L
.38608 1.30605 L
.38608 1.31793 L
F
.76546 1.49602 m
.76582 1.49613 L
.76582 1.5079 L
.75949 1.5079 L
.75949 1.49602 L
F
.4019 .55804 m
.40506 .55804 L
.40506 .56991 L
.39873 .56991 L
.39873 .55804 L
F
.73418 1.49824 m
.73855 1.49602 L
.74051 1.49602 L
.74051 1.5079 L
.73418 1.5079 L
F
.72443 .36807 m
.72152 .36128 L
.72152 .3562 L
.72785 .3562 L
.72785 .36807 L
F
.41139 .56991 m
.41772 .56991 L
.41772 .59366 L
.40506 .59366 L
.40506 .56991 L
F
.39873 .55804 m
.39873 .56991 L
.39241 .56991 L
.39241 .55804 L
.39557 .55804 L
F
.75316 .36807 m
.74684 .36807 L
.74684 .3562 L
.75 .3562 L
.75316 .3562 L
F
.41772 1.30605 m
.43038 1.30605 L
.43038 1.3298 L
.41772 1.3298 L
.41772 1.31793 L
F
.37342 .56991 m
.37975 .56991 L
.37975 .59366 L
.36709 .59366 L
.36709 .56991 L
F
.41139 1.35355 m
.41772 1.35355 L
.41772 1.37729 L
.40506 1.37729 L
.40506 1.35355 L
F
.38924 1.34167 m
.39241 1.34167 L
.39241 1.35355 L
.38608 1.35355 L
.38608 1.34167 L
F
.75949 1.56349 m
.75316 1.56347 L
.75316 1.55539 L
.75633 1.55539 L
.75949 1.55539 L
F
.75316 1.56347 m
.74684 1.5622 L
.74684 1.55539 L
.75 1.55539 L
.75316 1.55539 L
F
.77215 1.55956 m
.76582 1.56223 L
.76582 1.55539 L
.76899 1.55539 L
.77215 1.55539 L
F
.38608 1.34167 m
.38608 1.35355 L
.37975 1.35355 L
.37975 1.34167 L
.38291 1.34167 L
F
.39873 .56991 m
.40506 .56991 L
.40506 .59366 L
.39241 .59366 L
.39241 .56991 L
F
.38608 .56991 m
.39241 .56991 L
.39241 .59366 L
.37975 .59366 L
.37975 .56991 L
F
.76582 1.5079 m
.77215 1.5079 L
.77215 1.51977 L
.76899 1.51977 L
.76582 1.51977 L
F
.35443 .54617 m
.36709 .54617 L
.36709 .55804 L
.36709 .56991 L
.35443 .56991 L
F
.76582 1.5079 m
.76582 1.51977 L
.76266 1.51977 L
.75949 1.51977 L
.75949 1.5079 L
F
.59494 1.2553 m
.59465 1.25856 L
.58228 1.25856 L
.58228 1.23481 L
.59494 1.23481 L
F
.12015 1.18732 m
.11392 1.17681 L
.11392 1.16357 L
.12658 1.16357 L
.12658 1.18732 L
F
.73341 .32058 m
.73418 .32058 L
.73418 .33245 L
.72785 .33245 L
.72785 .32703 L
F
.212 1.30605 m
.20253 1.29415 L
.20253 1.28231 L
.21519 1.28231 L
.21519 1.30605 L
F
.77848 1.5079 m
.77848 1.51977 L
.77532 1.51977 L
.77215 1.51977 L
.77215 1.5079 L
F
.58228 1.31943 m
.57892 1.3298 L
.56962 1.3298 L
.56962 1.30605 L
.58228 1.30605 L
F
.34041 .45118 m
.34177 .45118 L
.34177 .47493 L
.32911 .47493 L
.32911 .45709 L
F
.23138 1.3298 m
.22785 1.32521 L
.22785 1.30605 L
.24051 1.30605 L
.24051 1.3298 L
F
.39873 1.35355 m
.40506 1.35355 L
.40506 1.37729 L
.39241 1.37729 L
.39241 1.35355 L
F
.78131 1.5079 m
.78481 1.51337 L
.78481 1.51977 L
.77848 1.51977 L
.77848 1.5079 L
F
.38608 1.35355 m
.39241 1.35355 L
.39241 1.37729 L
.37975 1.37729 L
.37975 1.35355 L
F
.38924 .55804 m
.39241 .55804 L
.39241 .56991 L
.38608 .56991 L
.38608 .55804 L
F
.38608 .55804 m
.38608 .56991 L
.37975 .56991 L
.37975 .55804 L
.38291 .55804 L
F
.10649 1.16357 m
.10127 1.15305 L
.10127 1.13983 L
.11392 1.13983 L
.11392 1.16357 L
F
.72152 1.51469 m
.72443 1.5079 L
.72785 1.5079 L
.72785 1.51977 L
.72152 1.51977 L
F
.41139 .55804 m
.40506 .55804 L
.40506 .5521 L
.40506 .54617 L
.41139 .54617 L
F
.19243 1.28231 m
.18987 1.27916 L
.18987 1.25856 L
.20253 1.25856 L
.20253 1.28231 L
F
.72785 1.53164 m
.73418 1.53164 L
.73418 1.53758 L
.73418 1.54352 L
.72785 1.54352 L
F
.23138 .54617 m
.24051 .54617 L
.24051 .56991 L
.22785 .56991 L
.22785 .55076 L
F
.39557 1.31793 m
.39241 1.31793 L
.39241 1.30605 L
.39873 1.30605 L
.39873 1.31793 L
F
.22785 .75989 m
.24051 .75989 L
.24051 .78363 L
.23418 .78363 L
.22785 .78363 L
F
.75949 1.49394 m
.75949 1.49602 L
.75316 1.49602 L
.74684 1.49602 L
.74684 1.49343 L
F
.40506 1.28231 m
.41772 1.28231 L
.41772 1.30605 L
.41139 1.30605 L
.40506 1.30605 L
F
.37975 .49867 m
.39241 .49867 L
.39241 .52242 L
.38608 .52242 L
.37975 .52242 L
F
.39241 1.28231 m
.40506 1.28231 L
.40506 1.30605 L
.39873 1.30605 L
.39241 1.30605 L
F
.41139 1.31793 m
.41139 1.3298 L
.40506 1.3298 L
.40506 1.32386 L
.40506 1.31793 L
F
.17285 1.25856 m
.16456 1.24824 L
.16456 1.23481 L
.17722 1.23481 L
.17722 1.25856 L
F
.77848 .33245 m
.78481 .33245 L
.78481 .34432 L
.77848 .34432 L
.77848 .33839 L
F
.77848 .34432 m
.78481 .34432 L
.78481 .3562 L
.77848 .3562 L
.77848 .35026 L
F
.73341 1.55539 m
.72785 1.54893 L
.72785 1.54352 L
.73418 1.54352 L
.73418 1.55539 L
F
.74051 1.5079 m
.74684 1.5079 L
.74684 1.51977 L
.74367 1.51977 L
.74051 1.51977 L
F
.75316 1.5079 m
.75316 1.51977 L
.75 1.51977 L
.74684 1.51977 L
.74684 1.5079 L
F
.37975 1.28231 m
.39241 1.28231 L
.39241 1.30605 L
.38608 1.30605 L
.37975 1.30605 L
F
.72152 .3393 m
.72372 .33245 L
.72785 .33245 L
.72785 .34432 L
.72152 .34432 L
F
.75316 1.5079 m
.75949 1.5079 L
.75949 1.51977 L
.75633 1.51977 L
.75316 1.51977 L
F
.72785 .33245 m
.73418 .33245 L
.73418 .33839 L
.73418 .34432 L
.72785 .34432 L
F
.74051 1.5079 m
.74051 1.51977 L
.73734 1.51977 L
.73418 1.51977 L
.73418 1.5079 L
F
.36709 .49867 m
.37975 .49867 L
.37975 .52242 L
.37342 .52242 L
.36709 .52242 L
F
.39241 .49867 m
.40506 .49867 L
.40506 .52242 L
.39873 .52242 L
.39241 .52242 L
F
.36709 1.28231 m
.37975 1.28231 L
.37975 1.30605 L
.37342 1.30605 L
.36709 1.30605 L
F
.19243 .59366 m
.20253 .59366 L
.20253 .61741 L
.18987 .61741 L
.18987 .59681 L
F
.74051 .36807 m
.73418 .36807 L
.73418 .3562 L
.73734 .3562 L
.74051 .3562 L
F
.39557 .53429 m
.39241 .53429 L
.39241 .52242 L
.39873 .52242 L
.39873 .53429 L
F
.37342 .54617 m
.37975 .54617 L
.37975 .5521 L
.37975 .55804 L
.37342 .55804 L
F
.72372 1.54352 m
.72152 1.53667 L
.72152 1.53164 L
.72785 1.53164 L
.72785 1.54352 L
F
.41772 .54617 m
.43038 .54617 L
.43038 .56991 L
.41772 .56991 L
.41772 .55804 L
F
.41772 1.3298 m
.43038 1.3298 L
.43038 1.35355 L
.41772 1.35355 L
.41772 1.34167 L
F
.212 .56991 m
.21519 .56991 L
.21519 .59366 L
.20253 .59366 L
.20253 .58182 L
F
.76582 .36807 m
.75949 .36807 L
.75949 .3562 L
.76266 .3562 L
.76582 .3562 L
F
.76582 1.56223 m
.75949 1.56349 L
.75949 1.55539 L
.76266 1.55539 L
.76582 1.55539 L
F
.39873 1.30605 m
.40506 1.30605 L
.40506 1.31793 L
.4019 1.31793 L
.39873 1.31793 L
F
.38608 1.30605 m
.39241 1.30605 L
.39241 1.31793 L
.38924 1.31793 L
.38608 1.31793 L
F
.35443 1.30605 m
.36709 1.30605 L
.36709 1.31793 L
.36709 1.3298 L
.35443 1.3298 L
F
.37342 .53429 m
.37975 .53429 L
.37975 .54023 L
.37975 .54617 L
.37342 .54617 L
F
.40506 .49867 m
.41772 .49867 L
.41772 .52242 L
.41139 .52242 L
.40506 .52242 L
F
.77848 .36807 m
.77215 .36807 L
.77215 .3562 L
.77532 .3562 L
.77848 .3562 L
F
.15366 1.23481 m
.1519 1.23254 L
.1519 1.21107 L
.16456 1.21107 L
.16456 1.23481 L
F
.38291 .53429 m
.37975 .53429 L
.37975 .52242 L
.38608 .52242 L
.38608 .53429 L
F
.41139 .53429 m
.41139 .54617 L
.40506 .54617 L
.40506 .54023 L
.40506 .53429 L
F
.4019 1.34167 m
.40506 1.34167 L
.40506 1.35355 L
.39873 1.35355 L
.39873 1.34167 L
F
.15366 .64115 m
.16456 .64115 L
.16456 .6649 L
.1519 .6649 L
.1519 .64342 L
F
.74684 1.5622 m
.74051 1.55968 L
.74051 1.55539 L
.74367 1.55539 L
.74684 1.55539 L
F
.39873 1.34167 m
.39873 1.35355 L
.39241 1.35355 L
.39241 1.34167 L
.39557 1.34167 L
F
.74051 1.55539 m
.73418 1.55539 L
.73418 1.54352 L
.73734 1.54352 L
.74051 1.54352 L
.74051 1.54945 L
F
.74051 .32651 m
.74051 .33245 L
.73734 .33245 L
.73418 .33245 L
.73418 .32058 L
.74051 .32058 L
F
.77848 .32061 m
.77848 .33245 L
.77532 .33245 L
.77215 .33245 L
.77215 .32651 L
.77215 .32058 L
.77845 .32058 L
F
.77845 1.55539 m
.77215 1.55539 L
.77215 1.54945 L
.77215 1.54352 L
.77532 1.54352 L
.77848 1.54352 L
.77848 1.55536 L
F
.5 g
.72443 .36807 m
.72152 .36807 L
.72152 .36128 L
F
.15366 1.23481 m
.1519 1.23481 L
.1519 1.23254 L
F
.73341 1.55539 m
.72785 1.55539 L
.72785 1.54893 L
F
.52483 .47493 m
.53165 .47493 L
.53165 .48079 L
F
.07841 1.09233 m
.07595 1.09233 L
.07595 1.08224 L
F
.72152 .3393 m
.72152 .33245 L
.72372 .33245 L
F
.56962 1.34967 m
.56962 1.35355 L
.56771 1.35355 L
F
.06382 .87862 m
.06329 .88914 L
.06329 .87862 L
F
.13595 .6649 m
.12658 .67875 L
.12658 .6649 L
F
.212 1.30605 m
.20253 1.30605 L
.20253 1.29415 L
F
.25129 1.35355 m
.24051 1.35355 L
.24051 1.34018 L
F
.09509 1.13983 m
.08861 1.13983 L
.08861 1.12365 L
F
.59494 1.2553 m
.59494 1.25856 L
.59465 1.25856 L
F
.78481 .3626 m
.78481 .36807 L
.78131 .36807 L
F
.13595 1.21107 m
.12658 1.21107 L
.12658 1.19722 L
F
.17285 1.25856 m
.16456 1.25856 L
.16456 1.24824 L
F
.48213 .45118 m
.49367 .45118 L
.49367 .45678 L
F
.56771 .52242 m
.56962 .52242 L
.56962 .52629 L
F
.19243 1.28231 m
.18987 1.28231 L
.18987 1.27916 L
F
.06382 .99735 m
.06329 .99735 L
.06329 .98683 L
F
.49367 1.41919 m
.49367 1.42478 L
.48213 1.42478 L
F
.34041 1.42478 m
.32911 1.42478 L
.32911 1.41887 L
F
.23138 1.3298 m
.22785 1.3298 L
.22785 1.32521 L
F
.34041 .45118 m
.32911 .45709 L
.32911 .45118 L
F
.58228 1.31943 m
.58228 1.3298 L
.57892 1.3298 L
F
.77845 1.55539 m
.77848 1.55536 L
.77848 1.55539 L
F
.10649 .71239 m
.10127 .72292 L
.10127 .71239 L
F
.30044 .47493 m
.29114 .48305 L
.29114 .47493 L
F
.25129 .52242 m
.24051 .53579 L
.24051 .52242 L
F
.73341 .32058 m
.72785 .32703 L
.72785 .32058 L
F
.27338 .49867 m
.26582 .5071 L
.26582 .49867 L
F
.77848 .32061 m
.77845 .32058 L
.77848 .32058 L
F
.23138 .54617 m
.22785 .55076 L
.22785 .54617 L
F
.19243 .59366 m
.18987 .59681 L
.18987 .59366 L
F
.17285 .61741 m
.16456 .62773 L
.16456 .61741 L
F
.27338 1.37729 m
.26582 1.37729 L
.26582 1.36887 L
F
.15366 .64115 m
.1519 .64342 L
.1519 .64115 L
F
.212 .56991 m
.20253 .58182 L
.20253 .56991 L
F
.12015 .68865 m
.11392 .69915 L
.11392 .68865 L
F
.10649 1.16357 m
.10127 1.16357 L
.10127 1.15305 L
F
.09509 .73614 m
.08861 .75232 L
.08861 .73614 L
F
.07841 .78363 m
.07595 .79373 L
.07595 .78363 L
F
.55696 1.36897 m
.55696 1.37729 L
.55078 1.37729 L
F
.72372 1.54352 m
.72152 1.54352 L
.72152 1.53667 L
F
.78131 1.5079 m
.78481 1.5079 L
.78481 1.51337 L
F
.57892 .54617 m
.58228 .54617 L
.58228 .55654 L
F
.53165 1.39518 m
.53165 1.40104 L
.52483 1.40104 L
F
.12015 1.18732 m
.11392 1.18732 L
.11392 1.17681 L
F
.76582 .37984 m
.76582 .37994 L
.76546 .37994 L
F
.76546 1.49602 m
.76582 1.49602 L
.76582 1.49613 L
F
.73855 .37994 m
.73418 .37994 L
.73418 .37773 L
F
.72152 1.51469 m
.72152 1.5079 L
.72443 1.5079 L
F
.30044 1.40104 m
.29114 1.40104 L
.29114 1.39292 L
F
.55078 .49867 m
.55696 .49867 L
.55696 .507 L
F
.59465 .61741 m
.59494 .61741 L
.59494 .62067 L
F
.73418 1.49824 m
.73418 1.49602 L
.73855 1.49602 L
F
.75949 .18997 m
.77215 .18997 L
.77215 .21372 L
.75949 .21372 L
F
.73418 .18997 m
.74684 .18997 L
.74684 .21372 L
.73418 .21372 L
F
.81013 .18997 m
.82278 .18997 L
.82278 .21372 L
.81013 .21372 L
F
.70886 .18997 m
.72152 .18997 L
.72152 .21372 L
.70886 .21372 L
F
.4557 .11873 m
.46835 .11873 L
.46835 .14248 L
.4557 .14248 L
F
.50633 .11873 m
.51899 .11873 L
.51899 .14248 L
.50633 .14248 L
F
.6962 .18997 m
.70886 .18997 L
.70886 .21372 L
.6962 .21372 L
F
.48101 .11873 m
.49367 .11873 L
.49367 .14248 L
.48101 .14248 L
F
.49367 .11873 m
.50633 .11873 L
.50633 .14248 L
.49367 .14248 L
F
.40506 .11873 m
.41772 .11873 L
.41772 .14248 L
.40506 .14248 L
F
.72152 .18997 m
.73418 .18997 L
.73418 .21372 L
.72152 .21372 L
F
.67089 .18997 m
.68354 .18997 L
.68354 .21372 L
.67089 .21372 L
F
.65823 .18997 m
.67089 .18997 L
.67089 .21372 L
.65823 .21372 L
F
.55696 .18997 m
.56962 .18997 L
.56962 .21372 L
.55696 .21372 L
F
.58228 .18997 m
.59494 .18997 L
.59494 .21372 L
.58228 .21372 L
F
.64557 .18997 m
.65823 .18997 L
.65823 .21372 L
.64557 .21372 L
F
.92405 .16622 m
.93671 .16622 L
.93671 .18997 L
.92405 .18997 L
F
.68354 .18997 m
.6962 .18997 L
.6962 .21372 L
.68354 .21372 L
F
.63291 .18997 m
.64557 .18997 L
.64557 .21372 L
.63291 .21372 L
F
.77215 .18997 m
.78481 .18997 L
.78481 .21372 L
.77215 .21372 L
F
.78481 .18997 m
.79747 .18997 L
.79747 .21372 L
.78481 .21372 L
F
.79747 .18997 m
.81013 .18997 L
.81013 .21372 L
.79747 .21372 L
F
.82278 .18997 m
.83544 .18997 L
.83544 .21372 L
.82278 .21372 L
F
.62025 .18997 m
.63291 .18997 L
.63291 .21372 L
.62025 .21372 L
F
.56962 .18997 m
.58228 .18997 L
.58228 .21372 L
.56962 .21372 L
F
.60759 .18997 m
.62025 .18997 L
.62025 .21372 L
.60759 .21372 L
F
.59494 .18997 m
.60759 .18997 L
.60759 .21372 L
.59494 .21372 L
F
.34177 .11873 m
.35443 .11873 L
.35443 .14248 L
.34177 .14248 L
F
.5443 .18997 m
.55696 .18997 L
.55696 .21372 L
.5443 .21372 L
F
.49367 .18997 m
.50633 .18997 L
.50633 .21372 L
.49367 .21372 L
F
.51899 .18997 m
.53165 .18997 L
.53165 .21372 L
.51899 .21372 L
F
.36709 .11873 m
.37975 .11873 L
.37975 .14248 L
.36709 .14248 L
F
.31646 .18997 m
.32911 .18997 L
.32911 .21372 L
.31646 .21372 L
F
.34177 .18997 m
.35443 .18997 L
.35443 .21372 L
.34177 .21372 L
F
.50633 .18997 m
.51899 .18997 L
.51899 .21372 L
.50633 .21372 L
F
.32911 .18997 m
.34177 .18997 L
.34177 .21372 L
.32911 .21372 L
F
.48101 .18997 m
.49367 .18997 L
.49367 .21372 L
.48101 .21372 L
F
.43038 .18997 m
.44304 .18997 L
.44304 .21372 L
.43038 .21372 L
F
.4557 .18997 m
.46835 .18997 L
.46835 .21372 L
.4557 .21372 L
F
.32911 .11873 m
.34177 .11873 L
.34177 .14248 L
.32911 .14248 L
F
.31646 .11873 m
.32911 .11873 L
.32911 .14248 L
.31646 .14248 L
F
.53165 .18997 m
.5443 .18997 L
.5443 .21372 L
.53165 .21372 L
F
.41772 .18997 m
.43038 .18997 L
.43038 .21372 L
.41772 .21372 L
F
.44304 .18997 m
.4557 .18997 L
.4557 .21372 L
.44304 .21372 L
F
.27848 .11873 m
.29114 .11873 L
.29114 .14248 L
.27848 .14248 L
F
.3038 .11873 m
.31646 .11873 L
.31646 .14248 L
.3038 .14248 L
F
.35443 .18997 m
.36709 .18997 L
.36709 .21372 L
.35443 .21372 L
F
.40506 .18997 m
.41772 .18997 L
.41772 .21372 L
.40506 .21372 L
F
.25316 .11873 m
.26582 .11873 L
.26582 .14248 L
.25316 .14248 L
F
.74684 .18997 m
.75949 .18997 L
.75949 .21372 L
.74684 .21372 L
F
.46835 .11873 m
.48101 .11873 L
.48101 .14248 L
.46835 .14248 L
F
.29114 .11873 m
.3038 .11873 L
.3038 .14248 L
.29114 .14248 L
F
.36709 .18997 m
.37975 .18997 L
.37975 .21372 L
.36709 .21372 L
F
.27848 .18997 m
.29114 .18997 L
.29114 .21372 L
.27848 .21372 L
F
.3038 .18997 m
.31646 .18997 L
.31646 .21372 L
.3038 .21372 L
F
.35443 .11873 m
.36709 .11873 L
.36709 .14248 L
.35443 .14248 L
F
.29114 .18997 m
.3038 .18997 L
.3038 .21372 L
.29114 .21372 L
F
.37975 .18997 m
.39241 .18997 L
.39241 .21372 L
.37975 .21372 L
F
.24051 .11873 m
.25316 .11873 L
.25316 .14248 L
.24051 .14248 L
F
.26582 .11873 m
.27848 .11873 L
.27848 .14248 L
.26582 .14248 L
F
.39241 .18997 m
.40506 .18997 L
.40506 .21372 L
.39241 .21372 L
F
.07595 .18997 m
.08861 .18997 L
.08861 .21372 L
.07595 .21372 L
F
.12658 .18997 m
.13924 .18997 L
.13924 .21372 L
.12658 .21372 L
F
.03797 .11873 m
.05063 .11873 L
.05063 .14248 L
.03797 .14248 L
F
.10127 .18997 m
.11392 .18997 L
.11392 .21372 L
.10127 .21372 L
F
.05063 .11873 m
.06329 .11873 L
.06329 .14248 L
.05063 .14248 L
F
.1519 .11873 m
.16456 .11873 L
.16456 .14248 L
.1519 .14248 L
F
.13924 .11873 m
.1519 .11873 L
.1519 .14248 L
.13924 .14248 L
F
.08861 .11873 m
.10127 .11873 L
.10127 .14248 L
.08861 .14248 L
F
.08861 .18997 m
.10127 .18997 L
.10127 .21372 L
.08861 .21372 L
F
.05063 .18997 m
.06329 .18997 L
.06329 .21372 L
.05063 .21372 L
F
.22785 .18997 m
.24051 .18997 L
.24051 .21372 L
.22785 .21372 L
F
.03797 .18997 m
.05063 .18997 L
.05063 .21372 L
.03797 .21372 L
F
.26582 .18997 m
.27848 .18997 L
.27848 .21372 L
.26582 .21372 L
F
.21519 .18997 m
.22785 .18997 L
.22785 .21372 L
.21519 .21372 L
F
.02532 .18997 m
.03797 .18997 L
.03797 .21372 L
.02532 .21372 L
F
.11392 .18997 m
.12658 .18997 L
.12658 .21372 L
.11392 .21372 L
F
.06329 .18997 m
.07595 .18997 L
.07595 .21372 L
.06329 .21372 L
F
.01266 .18997 m
.02532 .18997 L
.02532 .21372 L
.01266 .21372 L
F
.17722 .11873 m
.18987 .11873 L
.18987 .14248 L
.17722 .14248 L
F
.96203 .09499 m
.97468 .09499 L
.97468 .11873 L
.96203 .11873 L
F
.18987 .11873 m
.20253 .11873 L
.20253 .14248 L
.18987 .14248 L
F
.20253 .11873 m
.21519 .11873 L
.21519 .14248 L
.20253 .14248 L
F
.22785 .11873 m
.24051 .11873 L
.24051 .14248 L
.22785 .14248 L
F
0 .11873 m
.01266 .11873 L
.01266 .14248 L
0 .14248 L
F
.02532 .11873 m
.03797 .11873 L
.03797 .14248 L
.02532 .14248 L
F
.01266 .11873 m
.02532 .11873 L
.02532 .14248 L
.01266 .14248 L
F
.97468 .09499 m
.98734 .09499 L
.98734 .11873 L
.97468 .11873 L
F
.11392 .11873 m
.12658 .11873 L
.12658 .14248 L
.11392 .14248 L
F
.10127 .11873 m
.11392 .11873 L
.11392 .14248 L
.10127 .14248 L
F
.16456 .11873 m
.17722 .11873 L
.17722 .14248 L
.16456 .14248 L
F
.06329 .11873 m
.07595 .11873 L
.07595 .14248 L
.06329 .14248 L
F
.21519 .11873 m
.22785 .11873 L
.22785 .14248 L
.21519 .14248 L
F
.98734 .09499 m
1 .09499 L
1 .11873 L
.98734 .11873 L
F
.07595 .11873 m
.08861 .11873 L
.08861 .14248 L
.07595 .14248 L
F
.12658 .11873 m
.13924 .11873 L
.13924 .14248 L
.12658 .14248 L
F
.77215 .16622 m
.78481 .16622 L
.78481 .18997 L
.77215 .18997 L
F
.78481 .16622 m
.79747 .16622 L
.79747 .18997 L
.78481 .18997 L
F
.79747 .16622 m
.81013 .16622 L
.81013 .18997 L
.79747 .18997 L
F
.94937 .16622 m
.96203 .16622 L
.96203 .18997 L
.94937 .18997 L
F
.96203 .16622 m
.97468 .16622 L
.97468 .18997 L
.96203 .18997 L
F
.75949 .16622 m
.77215 .16622 L
.77215 .18997 L
.75949 .18997 L
F
.8481 .16622 m
.86076 .16622 L
.86076 .18997 L
.8481 .18997 L
F
.74684 .16622 m
.75949 .16622 L
.75949 .18997 L
.74684 .18997 L
F
.73418 .16622 m
.74684 .16622 L
.74684 .18997 L
.73418 .18997 L
F
.88608 .16622 m
.89873 .16622 L
.89873 .18997 L
.88608 .18997 L
F
.87342 .16622 m
.88608 .16622 L
.88608 .18997 L
.87342 .18997 L
F
.91139 .16622 m
.92405 .16622 L
.92405 .18997 L
.91139 .18997 L
F
.93671 .16622 m
.94937 .16622 L
.94937 .18997 L
.93671 .18997 L
F
.97468 .16622 m
.98734 .16622 L
.98734 .18997 L
.97468 .18997 L
F
0 .18997 m
.01266 .18997 L
.01266 .21372 L
0 .21372 L
F
.89873 .16622 m
.91139 .16622 L
.91139 .18997 L
.89873 .18997 L
F
.98734 .16622 m
1 .16622 L
1 .18997 L
.98734 .18997 L
F
.1519 .18997 m
.16456 .18997 L
.16456 .21372 L
.1519 .21372 L
F
.20253 .18997 m
.21519 .18997 L
.21519 .21372 L
.20253 .21372 L
F
.13924 .18997 m
.1519 .18997 L
.1519 .21372 L
.13924 .21372 L
F
.18987 .18997 m
.20253 .18997 L
.20253 .21372 L
.18987 .21372 L
F
.17722 .18997 m
.18987 .18997 L
.18987 .21372 L
.17722 .21372 L
F
.25316 .18997 m
.26582 .18997 L
.26582 .21372 L
.25316 .21372 L
F
.24051 .18997 m
.25316 .18997 L
.25316 .21372 L
.24051 .21372 L
F
.43038 .11873 m
.44304 .11873 L
.44304 .14248 L
.43038 .14248 L
F
.16456 .18997 m
.17722 .18997 L
.17722 .21372 L
.16456 .21372 L
F
.86076 .16622 m
.87342 .16622 L
.87342 .18997 L
.86076 .18997 L
F
.39241 .11873 m
.40506 .11873 L
.40506 .14248 L
.39241 .14248 L
F
.83544 .16622 m
.8481 .16622 L
.8481 .18997 L
.83544 .18997 L
F
.37975 .11873 m
.39241 .11873 L
.39241 .14248 L
.37975 .14248 L
F
.41772 .11873 m
.43038 .11873 L
.43038 .14248 L
.41772 .14248 L
F
.82278 .16622 m
.83544 .16622 L
.83544 .18997 L
.82278 .18997 L
F
.81013 .16622 m
.82278 .16622 L
.82278 .18997 L
.81013 .18997 L
F
.44304 .11873 m
.4557 .11873 L
.4557 .14248 L
.44304 .14248 L
F
.50633 .02375 m
.51899 .02375 L
.51899 .04749 L
.50633 .04749 L
F
.51899 .02375 m
.53165 .02375 L
.53165 .04749 L
.51899 .04749 L
F
.53165 .02375 m
.5443 .02375 L
.5443 .04749 L
.53165 .04749 L
F
.5443 .02375 m
.55696 .02375 L
.55696 .04749 L
.5443 .04749 L
F
.56962 .02375 m
.58228 .02375 L
.58228 .04749 L
.56962 .04749 L
F
.58228 .02375 m
.59494 .02375 L
.59494 .04749 L
.58228 .04749 L
F
.63291 .02375 m
.64557 .02375 L
.64557 .04749 L
.63291 .04749 L
F
.59494 .02375 m
.60759 .02375 L
.60759 .04749 L
.59494 .04749 L
F
.62025 .02375 m
.63291 .02375 L
.63291 .04749 L
.62025 .04749 L
F
.60759 .02375 m
.62025 .02375 L
.62025 .04749 L
.60759 .04749 L
F
.12658 .02375 m
.13924 .02375 L
.13924 .04749 L
.12658 .04749 L
F
.10127 .02375 m
.11392 .02375 L
.11392 .04749 L
.10127 .04749 L
F
.21519 .02375 m
.22785 .02375 L
.22785 .04749 L
.21519 .04749 L
F
.16456 .02375 m
.17722 .02375 L
.17722 .04749 L
.16456 .04749 L
F
.11392 .02375 m
.12658 .02375 L
.12658 .04749 L
.11392 .04749 L
F
.18987 .02375 m
.20253 .02375 L
.20253 .04749 L
.18987 .04749 L
F
.17722 .02375 m
.18987 .02375 L
.18987 .04749 L
.17722 .04749 L
F
.55696 .02375 m
.56962 .02375 L
.56962 .04749 L
.55696 .04749 L
F
.20253 .02375 m
.21519 .02375 L
.21519 .04749 L
.20253 .04749 L
F
.22785 .02375 m
.24051 .02375 L
.24051 .04749 L
.22785 .04749 L
F
.32911 .16622 m
.34177 .16622 L
.34177 .18997 L
.32911 .18997 L
F
.37975 .16622 m
.39241 .16622 L
.39241 .18997 L
.37975 .18997 L
F
.31646 .16622 m
.32911 .16622 L
.32911 .18997 L
.31646 .18997 L
F
.36709 .16622 m
.37975 .16622 L
.37975 .18997 L
.36709 .18997 L
F
.35443 .16622 m
.36709 .16622 L
.36709 .18997 L
.35443 .18997 L
F
.44304 .16622 m
.4557 .16622 L
.4557 .18997 L
.44304 .18997 L
F
.41772 .16622 m
.43038 .16622 L
.43038 .18997 L
.41772 .18997 L
F
.39241 .16622 m
.40506 .16622 L
.40506 .18997 L
.39241 .18997 L
F
.34177 .16622 m
.35443 .16622 L
.35443 .18997 L
.34177 .18997 L
F
.43038 .16622 m
.44304 .16622 L
.44304 .18997 L
.43038 .18997 L
F
.37975 .02375 m
.39241 .02375 L
.39241 .04749 L
.37975 .04749 L
F
.43038 .02375 m
.44304 .02375 L
.44304 .04749 L
.43038 .04749 L
F
.39241 .02375 m
.40506 .02375 L
.40506 .04749 L
.39241 .04749 L
F
.41772 .02375 m
.43038 .02375 L
.43038 .04749 L
.41772 .04749 L
F
.40506 .02375 m
.41772 .02375 L
.41772 .04749 L
.40506 .04749 L
F
.44304 .02375 m
.4557 .02375 L
.4557 .04749 L
.44304 .04749 L
F
.49367 .02375 m
.50633 .02375 L
.50633 .04749 L
.49367 .04749 L
F
.4557 .02375 m
.46835 .02375 L
.46835 .04749 L
.4557 .04749 L
F
.48101 .02375 m
.49367 .02375 L
.49367 .04749 L
.48101 .04749 L
F
.46835 .02375 m
.48101 .02375 L
.48101 .04749 L
.46835 .04749 L
F
.13924 .04749 m
.1519 .04749 L
.1519 .07124 L
.13924 .07124 L
F
.18987 .04749 m
.20253 .04749 L
.20253 .07124 L
.18987 .07124 L
F
.1519 .04749 m
.16456 .04749 L
.16456 .07124 L
.1519 .07124 L
F
.17722 .04749 m
.18987 .04749 L
.18987 .07124 L
.17722 .07124 L
F
.16456 .04749 m
.17722 .04749 L
.17722 .07124 L
.16456 .07124 L
F
.93671 .02375 m
.94937 .02375 L
.94937 .04749 L
.93671 .04749 L
F
.98734 .02375 m
1 .02375 L
1 .04749 L
.98734 .04749 L
F
.92405 .02375 m
.93671 .02375 L
.93671 .04749 L
.92405 .04749 L
F
.97468 .02375 m
.98734 .02375 L
.98734 .04749 L
.97468 .04749 L
F
.96203 .02375 m
.97468 .02375 L
.97468 .04749 L
.96203 .04749 L
F
.73418 .02375 m
.74684 .02375 L
.74684 .04749 L
.73418 .04749 L
F
.72152 .02375 m
.73418 .02375 L
.73418 .04749 L
.72152 .04749 L
F
.11392 .04749 m
.12658 .04749 L
.12658 .07124 L
.11392 .07124 L
F
.74684 .02375 m
.75949 .02375 L
.75949 .04749 L
.74684 .04749 L
F
.77215 .02375 m
.78481 .02375 L
.78481 .04749 L
.77215 .04749 L
F
.06329 .04749 m
.07595 .04749 L
.07595 .07124 L
.06329 .07124 L
F
.07595 .04749 m
.08861 .04749 L
.08861 .07124 L
.07595 .07124 L
F
.08861 .04749 m
.10127 .04749 L
.10127 .07124 L
.08861 .07124 L
F
.10127 .04749 m
.11392 .04749 L
.11392 .07124 L
.10127 .07124 L
F
.12658 .04749 m
.13924 .04749 L
.13924 .07124 L
.12658 .07124 L
F
.35443 .02375 m
.36709 .02375 L
.36709 .04749 L
.35443 .04749 L
F
.26582 .02375 m
.27848 .02375 L
.27848 .04749 L
.26582 .04749 L
F
.34177 .02375 m
.35443 .02375 L
.35443 .04749 L
.34177 .04749 L
F
.25316 .02375 m
.26582 .02375 L
.26582 .04749 L
.25316 .04749 L
F
.24051 .02375 m
.25316 .02375 L
.25316 .04749 L
.24051 .04749 L
F
.1519 .02375 m
.16456 .02375 L
.16456 .04749 L
.1519 .04749 L
F
.32911 .02375 m
.34177 .02375 L
.34177 .04749 L
.32911 .04749 L
F
.13924 .02375 m
.1519 .02375 L
.1519 .04749 L
.13924 .04749 L
F
.36709 .02375 m
.37975 .02375 L
.37975 .04749 L
.36709 .04749 L
F
.31646 .02375 m
.32911 .02375 L
.32911 .04749 L
.31646 .04749 L
F
.05063 .04749 m
.06329 .04749 L
.06329 .07124 L
.05063 .07124 L
F
.02532 .04749 m
.03797 .04749 L
.03797 .07124 L
.02532 .07124 L
F
0 .04749 m
.01266 .04749 L
.01266 .07124 L
0 .07124 L
F
.94937 .02375 m
.96203 .02375 L
.96203 .04749 L
.94937 .04749 L
F
.03797 .04749 m
.05063 .04749 L
.05063 .07124 L
.03797 .07124 L
F
.27848 .02375 m
.29114 .02375 L
.29114 .04749 L
.27848 .04749 L
F
.3038 .02375 m
.31646 .02375 L
.31646 .04749 L
.3038 .04749 L
F
.01266 .04749 m
.02532 .04749 L
.02532 .07124 L
.01266 .07124 L
F
.29114 .02375 m
.3038 .02375 L
.3038 .04749 L
.29114 .04749 L
F
.72152 .14248 m
.73418 .14248 L
.73418 .16622 L
.72152 .16622 L
F
.73418 .14248 m
.74684 .14248 L
.74684 .16622 L
.73418 .16622 L
F
.74684 .14248 m
.75949 .14248 L
.75949 .16622 L
.74684 .16622 L
F
.62025 .14248 m
.63291 .14248 L
.63291 .16622 L
.62025 .16622 L
F
.64557 .14248 m
.65823 .14248 L
.65823 .16622 L
.64557 .16622 L
F
.07595 .16622 m
.08861 .16622 L
.08861 .18997 L
.07595 .18997 L
F
.08861 .16622 m
.10127 .16622 L
.10127 .18997 L
.08861 .18997 L
F
.10127 .16622 m
.11392 .16622 L
.11392 .18997 L
.10127 .18997 L
F
.6962 .14248 m
.70886 .14248 L
.70886 .16622 L
.6962 .16622 L
F
.70886 .14248 m
.72152 .14248 L
.72152 .16622 L
.70886 .16622 L
F
.88608 .14248 m
.89873 .14248 L
.89873 .16622 L
.88608 .16622 L
F
.86076 .14248 m
.87342 .14248 L
.87342 .16622 L
.86076 .16622 L
F
.83544 .14248 m
.8481 .14248 L
.8481 .16622 L
.83544 .16622 L
F
.78481 .14248 m
.79747 .14248 L
.79747 .16622 L
.78481 .16622 L
F
.87342 .14248 m
.88608 .14248 L
.88608 .16622 L
.87342 .16622 L
F
.68354 .14248 m
.6962 .14248 L
.6962 .16622 L
.68354 .16622 L
F
.65823 .14248 m
.67089 .14248 L
.67089 .16622 L
.65823 .16622 L
F
.63291 .14248 m
.64557 .14248 L
.64557 .16622 L
.63291 .16622 L
F
.8481 .14248 m
.86076 .14248 L
.86076 .16622 L
.8481 .16622 L
F
.67089 .14248 m
.68354 .14248 L
.68354 .16622 L
.67089 .16622 L
F
.92405 .14248 m
.93671 .14248 L
.93671 .16622 L
.92405 .16622 L
F
.89873 .14248 m
.91139 .14248 L
.91139 .16622 L
.89873 .16622 L
F
.01266 .16622 m
.02532 .16622 L
.02532 .18997 L
.01266 .18997 L
F
.96203 .14248 m
.97468 .14248 L
.97468 .16622 L
.96203 .16622 L
F
.91139 .14248 m
.92405 .14248 L
.92405 .16622 L
.91139 .16622 L
F
.98734 .14248 m
1 .14248 L
1 .16622 L
.98734 .16622 L
F
.97468 .14248 m
.98734 .14248 L
.98734 .16622 L
.97468 .16622 L
F
.46835 .18997 m
.48101 .18997 L
.48101 .21372 L
.46835 .21372 L
F
0 .16622 m
.01266 .16622 L
.01266 .18997 L
0 .18997 L
F
.02532 .16622 m
.03797 .16622 L
.03797 .18997 L
.02532 .18997 L
F
.1519 .16622 m
.16456 .16622 L
.16456 .18997 L
.1519 .18997 L
F
.06329 .16622 m
.07595 .16622 L
.07595 .18997 L
.06329 .18997 L
F
.13924 .16622 m
.1519 .16622 L
.1519 .18997 L
.13924 .18997 L
F
.05063 .16622 m
.06329 .16622 L
.06329 .18997 L
.05063 .18997 L
F
.03797 .16622 m
.05063 .16622 L
.05063 .18997 L
.03797 .18997 L
F
.94937 .14248 m
.96203 .14248 L
.96203 .16622 L
.94937 .16622 L
F
.12658 .16622 m
.13924 .16622 L
.13924 .18997 L
.12658 .18997 L
F
.93671 .14248 m
.94937 .14248 L
.94937 .16622 L
.93671 .16622 L
F
.16456 .16622 m
.17722 .16622 L
.17722 .18997 L
.16456 .18997 L
F
.11392 .16622 m
.12658 .16622 L
.12658 .18997 L
.11392 .18997 L
F
.63291 .16622 m
.64557 .16622 L
.64557 .18997 L
.63291 .18997 L
F
.64557 .16622 m
.65823 .16622 L
.65823 .18997 L
.64557 .18997 L
F
.65823 .16622 m
.67089 .16622 L
.67089 .18997 L
.65823 .18997 L
F
.25316 .16622 m
.26582 .16622 L
.26582 .18997 L
.25316 .18997 L
F
.26582 .16622 m
.27848 .16622 L
.27848 .18997 L
.26582 .18997 L
F
.70886 .16622 m
.72152 .16622 L
.72152 .18997 L
.70886 .18997 L
F
.62025 .16622 m
.63291 .16622 L
.63291 .18997 L
.62025 .18997 L
F
.6962 .16622 m
.70886 .16622 L
.70886 .18997 L
.6962 .18997 L
F
.60759 .16622 m
.62025 .16622 L
.62025 .18997 L
.60759 .18997 L
F
.59494 .16622 m
.60759 .16622 L
.60759 .18997 L
.59494 .18997 L
F
.24051 .16622 m
.25316 .16622 L
.25316 .18997 L
.24051 .18997 L
F
.21519 .16622 m
.22785 .16622 L
.22785 .18997 L
.21519 .18997 L
F
.18987 .16622 m
.20253 .16622 L
.20253 .18997 L
.18987 .18997 L
F
.40506 .16622 m
.41772 .16622 L
.41772 .18997 L
.40506 .18997 L
F
.22785 .16622 m
.24051 .16622 L
.24051 .18997 L
.22785 .18997 L
F
.27848 .16622 m
.29114 .16622 L
.29114 .18997 L
.27848 .18997 L
F
.29114 .16622 m
.3038 .16622 L
.3038 .18997 L
.29114 .18997 L
F
.3038 .16622 m
.31646 .16622 L
.31646 .18997 L
.3038 .18997 L
F
.17722 .16622 m
.18987 .16622 L
.18987 .18997 L
.17722 .18997 L
F
.20253 .16622 m
.21519 .16622 L
.21519 .18997 L
.20253 .18997 L
F
.53165 .16622 m
.5443 .16622 L
.5443 .18997 L
.53165 .18997 L
F
.58228 .16622 m
.59494 .16622 L
.59494 .18997 L
.58228 .18997 L
F
.5443 .16622 m
.55696 .16622 L
.55696 .18997 L
.5443 .18997 L
F
.56962 .16622 m
.58228 .16622 L
.58228 .18997 L
.56962 .18997 L
F
.55696 .16622 m
.56962 .16622 L
.56962 .18997 L
.55696 .18997 L
F
.77215 .14248 m
.78481 .14248 L
.78481 .16622 L
.77215 .16622 L
F
.82278 .14248 m
.83544 .14248 L
.83544 .16622 L
.82278 .16622 L
F
.75949 .14248 m
.77215 .14248 L
.77215 .16622 L
.75949 .16622 L
F
.81013 .14248 m
.82278 .14248 L
.82278 .16622 L
.81013 .16622 L
F
.79747 .14248 m
.81013 .14248 L
.81013 .16622 L
.79747 .16622 L
F
.50633 .16622 m
.51899 .16622 L
.51899 .18997 L
.50633 .18997 L
F
.68354 .16622 m
.6962 .16622 L
.6962 .18997 L
.68354 .18997 L
F
.49367 .16622 m
.50633 .16622 L
.50633 .18997 L
.49367 .18997 L
F
.72152 .16622 m
.73418 .16622 L
.73418 .18997 L
.72152 .18997 L
F
.67089 .16622 m
.68354 .16622 L
.68354 .18997 L
.67089 .18997 L
F
.4557 .16622 m
.46835 .16622 L
.46835 .18997 L
.4557 .18997 L
F
.48101 .16622 m
.49367 .16622 L
.49367 .18997 L
.48101 .18997 L
F
.51899 .16622 m
.53165 .16622 L
.53165 .18997 L
.51899 .18997 L
F
.46835 .16622 m
.48101 .16622 L
.48101 .18997 L
.46835 .18997 L
F
.98734 .87862 m
1 .87862 L
1 .90236 L
.98734 .90236 L
F
.03797 .90236 m
.05063 .90236 L
.05063 .92611 L
.03797 .92611 L
F
.97468 .87862 m
.98734 .87862 L
.98734 .90236 L
.97468 .90236 L
F
.02532 .90236 m
.03797 .90236 L
.03797 .92611 L
.02532 .92611 L
F
.01266 .90236 m
.02532 .90236 L
.02532 .92611 L
.01266 .92611 L
F
.64557 .90236 m
.65823 .90236 L
.65823 .92611 L
.64557 .92611 L
F
.62025 .90236 m
.63291 .90236 L
.63291 .92611 L
.62025 .92611 L
F
.59911 .90236 m
.60759 .90236 L
.60759 .92611 L
.59911 .92611 L
F
0 .90236 m
.01266 .90236 L
.01266 .92611 L
0 .92611 L
F
.63291 .90236 m
.64557 .90236 L
.64557 .92611 L
.63291 .92611 L
F
.03797 .87862 m
.05063 .87862 L
.05063 .90236 L
.03797 .90236 L
F
.01266 .87862 m
.02532 .87862 L
.02532 .90236 L
.01266 .90236 L
F
.67089 .87862 m
.68354 .87862 L
.68354 .90236 L
.67089 .90236 L
F
.62025 .87862 m
.63291 .87862 L
.63291 .90236 L
.62025 .90236 L
F
.02532 .87862 m
.03797 .87862 L
.03797 .90236 L
.02532 .90236 L
F
.63291 .87862 m
.64557 .87862 L
.64557 .90236 L
.63291 .90236 L
F
.64557 .87862 m
.65823 .87862 L
.65823 .90236 L
.64557 .90236 L
F
.65823 .87862 m
.67089 .87862 L
.67089 .90236 L
.65823 .90236 L
F
.68354 .87862 m
.6962 .87862 L
.6962 .90236 L
.68354 .90236 L
F
.03797 .9736 m
.05063 .9736 L
.05063 .99735 L
.03797 .99735 L
F
.92405 .87862 m
.93671 .87862 L
.93671 .90236 L
.92405 .90236 L
F
.02532 .9736 m
.03797 .9736 L
.03797 .99735 L
.02532 .99735 L
F
.96203 .87862 m
.97468 .87862 L
.97468 .90236 L
.96203 .90236 L
F
.91139 .87862 m
.92405 .87862 L
.92405 .90236 L
.91139 .90236 L
F
.63291 .9736 m
.64557 .9736 L
.64557 .99735 L
.63291 .99735 L
F
.01266 .9736 m
.02532 .9736 L
.02532 .99735 L
.01266 .99735 L
F
.62025 .9736 m
.63291 .9736 L
.63291 .99735 L
.62025 .99735 L
F
0 .9736 m
.01266 .9736 L
.01266 .99735 L
0 .99735 L
F
.98734 .94986 m
1 .94986 L
1 .9736 L
.98734 .9736 L
F
.89873 .87862 m
.91139 .87862 L
.91139 .90236 L
.89873 .90236 L
F
.87342 .87862 m
.88608 .87862 L
.88608 .90236 L
.87342 .90236 L
F
.8481 .87862 m
.86076 .87862 L
.86076 .90236 L
.8481 .90236 L
F
.60759 .90236 m
.62025 .90236 L
.62025 .92611 L
.60759 .92611 L
F
.88608 .87862 m
.89873 .87862 L
.89873 .90236 L
.88608 .90236 L
F
.94937 .87862 m
.96203 .87862 L
.96203 .90236 L
.94937 .90236 L
F
.93671 .87862 m
.94937 .87862 L
.94937 .90236 L
.93671 .90236 L
F
.83544 .87862 m
.8481 .87862 L
.8481 .90236 L
.83544 .90236 L
F
.86076 .87862 m
.87342 .87862 L
.87342 .90236 L
.86076 .90236 L
F
.70886 .92611 m
.72152 .92611 L
.72152 .94986 L
.70886 .94986 L
F
.6962 .92611 m
.70886 .92611 L
.70886 .94986 L
.6962 .94986 L
F
.98734 .90236 m
1 .90236 L
1 .92611 L
.98734 .92611 L
F
.72152 .92611 m
.73418 .92611 L
.73418 .94986 L
.72152 .94986 L
F
.74684 .92611 m
.75949 .92611 L
.75949 .94986 L
.74684 .94986 L
F
.93671 .90236 m
.94937 .90236 L
.94937 .92611 L
.93671 .92611 L
F
.94937 .90236 m
.96203 .90236 L
.96203 .92611 L
.94937 .92611 L
F
.96203 .90236 m
.97468 .90236 L
.97468 .92611 L
.96203 .92611 L
F
.97468 .90236 m
.98734 .90236 L
.98734 .92611 L
.97468 .92611 L
F
0 .92611 m
.01266 .92611 L
.01266 .94986 L
0 .94986 L
F
.65823 .92611 m
.67089 .92611 L
.67089 .94986 L
.65823 .94986 L
F
.67089 .92611 m
.68354 .92611 L
.68354 .94986 L
.67089 .94986 L
F
.68354 .92611 m
.6962 .92611 L
.6962 .94986 L
.68354 .94986 L
F
.73418 .90236 m
.74684 .90236 L
.74684 .92611 L
.73418 .92611 L
F
.74684 .90236 m
.75949 .90236 L
.75949 .92611 L
.74684 .92611 L
F
.64557 .92611 m
.65823 .92611 L
.65823 .94986 L
.64557 .94986 L
F
.73418 .92611 m
.74684 .92611 L
.74684 .94986 L
.73418 .94986 L
F
.63291 .92611 m
.64557 .92611 L
.64557 .94986 L
.63291 .94986 L
F
.62025 .92611 m
.63291 .92611 L
.63291 .94986 L
.62025 .94986 L
F
.79747 .87862 m
.81013 .87862 L
.81013 .90236 L
.79747 .90236 L
F
.81013 .87862 m
.82278 .87862 L
.82278 .90236 L
.81013 .90236 L
F
.82278 .87862 m
.83544 .87862 L
.83544 .90236 L
.82278 .90236 L
F
.6962 .87862 m
.70886 .87862 L
.70886 .90236 L
.6962 .90236 L
F
.72152 .87862 m
.73418 .87862 L
.73418 .90236 L
.72152 .90236 L
F
.60759 .87862 m
.62025 .87862 L
.62025 .90236 L
.60759 .90236 L
F
.59911 .87862 m
.60759 .87862 L
.60759 .90236 L
.59911 .90236 L
F
.77215 .87862 m
.78481 .87862 L
.78481 .90236 L
.77215 .90236 L
F
.78481 .87862 m
.79747 .87862 L
.79747 .90236 L
.78481 .90236 L
F
.01266 .92611 m
.02532 .92611 L
.02532 .94986 L
.01266 .94986 L
F
.60759 .92611 m
.62025 .92611 L
.62025 .94986 L
.60759 .94986 L
F
.02532 .92611 m
.03797 .92611 L
.03797 .94986 L
.02532 .94986 L
F
.59911 .92611 m
.60759 .92611 L
.60759 .94986 L
.59911 .94986 L
F
.03797 .92611 m
.05063 .92611 L
.05063 .94986 L
.03797 .94986 L
F
.75949 .87862 m
.77215 .87862 L
.77215 .90236 L
.75949 .90236 L
F
.70886 .87862 m
.72152 .87862 L
.72152 .90236 L
.70886 .90236 L
F
.74684 .87862 m
.75949 .87862 L
.75949 .90236 L
.74684 .90236 L
F
.73418 .87862 m
.74684 .87862 L
.74684 .90236 L
.73418 .90236 L
F
.82278 .92611 m
.83544 .92611 L
.83544 .94986 L
.82278 .94986 L
F
.79747 .92611 m
.81013 .92611 L
.81013 .94986 L
.79747 .94986 L
F
.77215 .92611 m
.78481 .92611 L
.78481 .94986 L
.77215 .94986 L
F
.98734 .92611 m
1 .92611 L
1 .94986 L
.98734 .94986 L
F
.81013 .92611 m
.82278 .92611 L
.82278 .94986 L
.81013 .94986 L
F
.86076 .92611 m
.87342 .92611 L
.87342 .94986 L
.86076 .94986 L
F
.87342 .92611 m
.88608 .92611 L
.88608 .94986 L
.87342 .94986 L
F
.88608 .92611 m
.89873 .92611 L
.89873 .94986 L
.88608 .94986 L
F
.75949 .92611 m
.77215 .92611 L
.77215 .94986 L
.75949 .94986 L
F
.78481 .92611 m
.79747 .92611 L
.79747 .94986 L
.78481 .94986 L
F
.89873 .92611 m
.91139 .92611 L
.91139 .94986 L
.89873 .94986 L
F
.91139 .92611 m
.92405 .92611 L
.92405 .94986 L
.91139 .94986 L
F
.92405 .92611 m
.93671 .92611 L
.93671 .94986 L
.92405 .94986 L
F
.93671 .92611 m
.94937 .92611 L
.94937 .94986 L
.93671 .94986 L
F
.96203 .92611 m
.97468 .92611 L
.97468 .94986 L
.96203 .94986 L
F
.02532 .94986 m
.03797 .94986 L
.03797 .9736 L
.02532 .9736 L
F
.97468 .92611 m
.98734 .92611 L
.98734 .94986 L
.97468 .94986 L
F
.01266 .94986 m
.02532 .94986 L
.02532 .9736 L
.01266 .9736 L
F
0 .94986 m
.01266 .94986 L
.01266 .9736 L
0 .9736 L
F
.79747 .94986 m
.81013 .94986 L
.81013 .9736 L
.79747 .9736 L
F
.78481 .94986 m
.79747 .94986 L
.79747 .9736 L
.78481 .9736 L
F
.62025 .94986 m
.63291 .94986 L
.63291 .9736 L
.62025 .9736 L
F
.81013 .94986 m
.82278 .94986 L
.82278 .9736 L
.81013 .9736 L
F
.83544 .94986 m
.8481 .94986 L
.8481 .9736 L
.83544 .9736 L
F
.59911 .94986 m
.60759 .94986 L
.60759 .9736 L
.59911 .9736 L
F
.03797 .94986 m
.05063 .94986 L
.05063 .9736 L
.03797 .9736 L
F
.68354 .94986 m
.6962 .94986 L
.6962 .9736 L
.68354 .9736 L
F
.60759 .94986 m
.62025 .94986 L
.62025 .9736 L
.60759 .9736 L
F
.63291 .94986 m
.64557 .94986 L
.64557 .9736 L
.63291 .9736 L
F
.74684 .94986 m
.75949 .94986 L
.75949 .9736 L
.74684 .9736 L
F
.75949 .94986 m
.77215 .94986 L
.77215 .9736 L
.75949 .9736 L
F
.77215 .94986 m
.78481 .94986 L
.78481 .9736 L
.77215 .9736 L
F
.83544 .92611 m
.8481 .92611 L
.8481 .94986 L
.83544 .94986 L
F
.8481 .92611 m
.86076 .92611 L
.86076 .94986 L
.8481 .94986 L
F
.73418 .94986 m
.74684 .94986 L
.74684 .9736 L
.73418 .9736 L
F
.82278 .94986 m
.83544 .94986 L
.83544 .9736 L
.82278 .9736 L
F
.72152 .94986 m
.73418 .94986 L
.73418 .9736 L
.72152 .9736 L
F
.70886 .94986 m
.72152 .94986 L
.72152 .9736 L
.70886 .9736 L
F
.92405 .94986 m
.93671 .94986 L
.93671 .9736 L
.92405 .9736 L
F
.97468 .94986 m
.98734 .94986 L
.98734 .9736 L
.97468 .9736 L
F
.93671 .94986 m
.94937 .94986 L
.94937 .9736 L
.93671 .9736 L
F
.96203 .94986 m
.97468 .94986 L
.97468 .9736 L
.96203 .9736 L
F
.94937 .94986 m
.96203 .94986 L
.96203 .9736 L
.94937 .9736 L
F
.81013 .9736 m
.82278 .9736 L
.82278 .99735 L
.81013 .99735 L
F
.86076 .9736 m
.87342 .9736 L
.87342 .99735 L
.86076 .99735 L
F
.79747 .9736 m
.81013 .9736 L
.81013 .99735 L
.79747 .99735 L
F
.8481 .9736 m
.86076 .9736 L
.86076 .99735 L
.8481 .99735 L
F
.83544 .9736 m
.8481 .9736 L
.8481 .99735 L
.83544 .99735 L
F
.89873 .94986 m
.91139 .94986 L
.91139 .9736 L
.89873 .9736 L
F
.60759 .9736 m
.62025 .9736 L
.62025 .99735 L
.60759 .99735 L
F
.88608 .94986 m
.89873 .94986 L
.89873 .9736 L
.88608 .9736 L
F
.64557 .9736 m
.65823 .9736 L
.65823 .99735 L
.64557 .99735 L
F
.59911 .9736 m
.60759 .9736 L
.60759 .99735 L
.59911 .99735 L
F
.8481 .94986 m
.86076 .94986 L
.86076 .9736 L
.8481 .9736 L
F
.87342 .94986 m
.88608 .94986 L
.88608 .9736 L
.87342 .9736 L
F
.91139 .94986 m
.92405 .94986 L
.92405 .9736 L
.91139 .9736 L
F
.86076 .94986 m
.87342 .94986 L
.87342 .9736 L
.86076 .9736 L
F
.77215 .9736 m
.78481 .9736 L
.78481 .99735 L
.77215 .99735 L
F
.68354 .9736 m
.6962 .9736 L
.6962 .99735 L
.68354 .99735 L
F
.75949 .9736 m
.77215 .9736 L
.77215 .99735 L
.75949 .99735 L
F
.67089 .9736 m
.68354 .9736 L
.68354 .99735 L
.67089 .99735 L
F
.65823 .9736 m
.67089 .9736 L
.67089 .99735 L
.65823 .99735 L
F
.74684 .9736 m
.75949 .9736 L
.75949 .99735 L
.74684 .99735 L
F
.94937 .92611 m
.96203 .92611 L
.96203 .94986 L
.94937 .94986 L
F
.78481 .9736 m
.79747 .9736 L
.79747 .99735 L
.78481 .99735 L
F
.73418 .9736 m
.74684 .9736 L
.74684 .99735 L
.73418 .99735 L
F
.92405 .9736 m
.93671 .9736 L
.93671 .99735 L
.92405 .99735 L
F
.89873 .9736 m
.91139 .9736 L
.91139 .99735 L
.89873 .99735 L
F
.87342 .9736 m
.88608 .9736 L
.88608 .99735 L
.87342 .99735 L
F
.82278 .9736 m
.83544 .9736 L
.83544 .99735 L
.82278 .99735 L
F
.91139 .9736 m
.92405 .9736 L
.92405 .99735 L
.91139 .99735 L
F
.6962 .9736 m
.70886 .9736 L
.70886 .99735 L
.6962 .99735 L
F
.72152 .9736 m
.73418 .9736 L
.73418 .99735 L
.72152 .99735 L
F
.88608 .9736 m
.89873 .9736 L
.89873 .99735 L
.88608 .99735 L
F
.70886 .9736 m
.72152 .9736 L
.72152 .99735 L
.70886 .99735 L
F
.31646 .14248 m
.32911 .14248 L
.32911 .16622 L
.31646 .16622 L
F
.22785 .14248 m
.24051 .14248 L
.24051 .16622 L
.22785 .16622 L
F
.3038 .14248 m
.31646 .14248 L
.31646 .16622 L
.3038 .16622 L
F
.21519 .14248 m
.22785 .14248 L
.22785 .16622 L
.21519 .16622 L
F
.20253 .14248 m
.21519 .14248 L
.21519 .16622 L
.20253 .16622 L
F
.11392 .14248 m
.12658 .14248 L
.12658 .16622 L
.11392 .16622 L
F
.29114 .14248 m
.3038 .14248 L
.3038 .16622 L
.29114 .16622 L
F
.10127 .14248 m
.11392 .14248 L
.11392 .16622 L
.10127 .16622 L
F
.32911 .14248 m
.34177 .14248 L
.34177 .16622 L
.32911 .16622 L
F
.27848 .14248 m
.29114 .14248 L
.29114 .16622 L
.27848 .16622 L
F
.81013 .09499 m
.82278 .09499 L
.82278 .11873 L
.81013 .11873 L
F
.78481 .09499 m
.79747 .09499 L
.79747 .11873 L
.78481 .11873 L
F
.75949 .09499 m
.77215 .09499 L
.77215 .11873 L
.75949 .11873 L
F
.70886 .09499 m
.72152 .09499 L
.72152 .11873 L
.70886 .11873 L
F
.79747 .09499 m
.81013 .09499 L
.81013 .11873 L
.79747 .11873 L
F
.24051 .14248 m
.25316 .14248 L
.25316 .16622 L
.24051 .16622 L
F
.26582 .14248 m
.27848 .14248 L
.27848 .16622 L
.26582 .16622 L
F
.77215 .09499 m
.78481 .09499 L
.78481 .11873 L
.77215 .11873 L
F
.25316 .14248 m
.26582 .14248 L
.26582 .16622 L
.25316 .16622 L
F
.49367 .14248 m
.50633 .14248 L
.50633 .16622 L
.49367 .16622 L
F
.5443 .14248 m
.55696 .14248 L
.55696 .16622 L
.5443 .16622 L
F
.48101 .14248 m
.49367 .14248 L
.49367 .16622 L
.48101 .16622 L
F
.53165 .14248 m
.5443 .14248 L
.5443 .16622 L
.53165 .16622 L
F
.51899 .14248 m
.53165 .14248 L
.53165 .16622 L
.51899 .16622 L
F
.60759 .14248 m
.62025 .14248 L
.62025 .16622 L
.60759 .16622 L
F
.58228 .14248 m
.59494 .14248 L
.59494 .16622 L
.58228 .16622 L
F
.55696 .14248 m
.56962 .14248 L
.56962 .16622 L
.55696 .16622 L
F
.50633 .14248 m
.51899 .14248 L
.51899 .16622 L
.50633 .16622 L
F
.59494 .14248 m
.60759 .14248 L
.60759 .16622 L
.59494 .16622 L
F
.08861 .14248 m
.10127 .14248 L
.10127 .16622 L
.08861 .16622 L
F
.06329 .14248 m
.07595 .14248 L
.07595 .16622 L
.06329 .16622 L
F
.17722 .14248 m
.18987 .14248 L
.18987 .16622 L
.17722 .16622 L
F
.12658 .14248 m
.13924 .14248 L
.13924 .16622 L
.12658 .16622 L
F
.07595 .14248 m
.08861 .14248 L
.08861 .16622 L
.07595 .16622 L
F
.13924 .14248 m
.1519 .14248 L
.1519 .16622 L
.13924 .16622 L
F
.1519 .14248 m
.16456 .14248 L
.16456 .16622 L
.1519 .16622 L
F
.16456 .14248 m
.17722 .14248 L
.17722 .16622 L
.16456 .16622 L
F
.18987 .14248 m
.20253 .14248 L
.20253 .16622 L
.18987 .16622 L
F
.4557 .09499 m
.46835 .09499 L
.46835 .11873 L
.4557 .11873 L
F
.46835 .09499 m
.48101 .09499 L
.48101 .11873 L
.46835 .11873 L
F
.48101 .09499 m
.49367 .09499 L
.49367 .11873 L
.48101 .11873 L
F
.63291 .09499 m
.64557 .09499 L
.64557 .11873 L
.63291 .11873 L
F
.64557 .09499 m
.65823 .09499 L
.65823 .11873 L
.64557 .11873 L
F
.53165 .09499 m
.5443 .09499 L
.5443 .11873 L
.53165 .11873 L
F
.44304 .09499 m
.4557 .09499 L
.4557 .11873 L
.44304 .11873 L
F
.51899 .09499 m
.53165 .09499 L
.53165 .11873 L
.51899 .11873 L
F
.43038 .09499 m
.44304 .09499 L
.44304 .11873 L
.43038 .11873 L
F
.41772 .09499 m
.43038 .09499 L
.43038 .11873 L
.41772 .11873 L
F
.56962 .09499 m
.58228 .09499 L
.58228 .11873 L
.56962 .11873 L
F
.62025 .09499 m
.63291 .09499 L
.63291 .11873 L
.62025 .11873 L
F
.55696 .09499 m
.56962 .09499 L
.56962 .11873 L
.55696 .11873 L
F
.60759 .09499 m
.62025 .09499 L
.62025 .11873 L
.60759 .11873 L
F
.59494 .09499 m
.60759 .09499 L
.60759 .11873 L
.59494 .11873 L
F
.65823 .09499 m
.67089 .09499 L
.67089 .11873 L
.65823 .11873 L
F
.68354 .09499 m
.6962 .09499 L
.6962 .11873 L
.68354 .11873 L
F
.58228 .09499 m
.59494 .09499 L
.59494 .11873 L
.58228 .11873 L
F
.67089 .09499 m
.68354 .09499 L
.68354 .11873 L
.67089 .11873 L
F
.89873 .09499 m
.91139 .09499 L
.91139 .11873 L
.89873 .11873 L
F
.94937 .09499 m
.96203 .09499 L
.96203 .11873 L
.94937 .11873 L
F
.91139 .09499 m
.92405 .09499 L
.92405 .11873 L
.91139 .11873 L
F
.93671 .09499 m
.94937 .09499 L
.94937 .11873 L
.93671 .11873 L
F
.92405 .09499 m
.93671 .09499 L
.93671 .11873 L
.92405 .11873 L
F
.74684 .09499 m
.75949 .09499 L
.75949 .11873 L
.74684 .11873 L
F
.6962 .09499 m
.70886 .09499 L
.70886 .11873 L
.6962 .11873 L
F
.73418 .09499 m
.74684 .09499 L
.74684 .11873 L
.73418 .11873 L
F
.72152 .09499 m
.73418 .09499 L
.73418 .11873 L
.72152 .11873 L
F
.87342 .09499 m
.88608 .09499 L
.88608 .11873 L
.87342 .11873 L
F
.50633 .09499 m
.51899 .09499 L
.51899 .11873 L
.50633 .11873 L
F
.86076 .09499 m
.87342 .09499 L
.87342 .11873 L
.86076 .11873 L
F
.5443 .09499 m
.55696 .09499 L
.55696 .11873 L
.5443 .11873 L
F
.49367 .09499 m
.50633 .09499 L
.50633 .11873 L
.49367 .11873 L
F
.82278 .09499 m
.83544 .09499 L
.83544 .11873 L
.82278 .11873 L
F
.8481 .09499 m
.86076 .09499 L
.86076 .11873 L
.8481 .11873 L
F
.88608 .09499 m
.89873 .09499 L
.89873 .11873 L
.88608 .11873 L
F
.83544 .09499 m
.8481 .09499 L
.8481 .11873 L
.83544 .11873 L
F
.81013 .11873 m
.82278 .11873 L
.82278 .14248 L
.81013 .14248 L
F
.79747 .11873 m
.81013 .11873 L
.81013 .14248 L
.79747 .14248 L
F
.89873 .11873 m
.91139 .11873 L
.91139 .14248 L
.89873 .14248 L
F
.82278 .11873 m
.83544 .11873 L
.83544 .14248 L
.82278 .14248 L
F
.8481 .11873 m
.86076 .11873 L
.86076 .14248 L
.8481 .14248 L
F
.87342 .11873 m
.88608 .11873 L
.88608 .14248 L
.87342 .14248 L
F
.86076 .11873 m
.87342 .11873 L
.87342 .14248 L
.86076 .14248 L
F
.8481 .90236 m
.86076 .90236 L
.86076 .92611 L
.8481 .92611 L
F
.88608 .11873 m
.89873 .11873 L
.89873 .14248 L
.88608 .14248 L
F
.91139 .11873 m
.92405 .11873 L
.92405 .14248 L
.91139 .14248 L
F
.03797 .14248 m
.05063 .14248 L
.05063 .16622 L
.03797 .16622 L
F
.94937 .11873 m
.96203 .11873 L
.96203 .14248 L
.94937 .14248 L
F
.02532 .14248 m
.03797 .14248 L
.03797 .16622 L
.02532 .16622 L
F
.93671 .11873 m
.94937 .11873 L
.94937 .14248 L
.93671 .14248 L
F
.92405 .11873 m
.93671 .11873 L
.93671 .14248 L
.92405 .14248 L
F
.01266 .14248 m
.02532 .14248 L
.02532 .16622 L
.01266 .16622 L
F
.83544 .11873 m
.8481 .11873 L
.8481 .14248 L
.83544 .14248 L
F
.05063 .14248 m
.06329 .14248 L
.06329 .16622 L
.05063 .16622 L
F
0 .14248 m
.01266 .14248 L
.01266 .16622 L
0 .16622 L
F
.72152 .90236 m
.73418 .90236 L
.73418 .92611 L
.72152 .92611 L
F
.6962 .90236 m
.70886 .90236 L
.70886 .92611 L
.6962 .92611 L
F
.67089 .90236 m
.68354 .90236 L
.68354 .92611 L
.67089 .92611 L
F
.88608 .90236 m
.89873 .90236 L
.89873 .92611 L
.88608 .92611 L
F
.70886 .90236 m
.72152 .90236 L
.72152 .92611 L
.70886 .92611 L
F
.75949 .90236 m
.77215 .90236 L
.77215 .92611 L
.75949 .92611 L
F
.77215 .90236 m
.78481 .90236 L
.78481 .92611 L
.77215 .92611 L
F
.78481 .90236 m
.79747 .90236 L
.79747 .92611 L
.78481 .92611 L
F
.65823 .90236 m
.67089 .90236 L
.67089 .92611 L
.65823 .92611 L
F
.68354 .90236 m
.6962 .90236 L
.6962 .92611 L
.68354 .92611 L
F
.79747 .90236 m
.81013 .90236 L
.81013 .92611 L
.79747 .92611 L
F
.81013 .90236 m
.82278 .90236 L
.82278 .92611 L
.81013 .92611 L
F
.82278 .90236 m
.83544 .90236 L
.83544 .92611 L
.82278 .92611 L
F
.83544 .90236 m
.8481 .90236 L
.8481 .92611 L
.83544 .92611 L
F
.86076 .90236 m
.87342 .90236 L
.87342 .92611 L
.86076 .92611 L
F
.92405 .90236 m
.93671 .90236 L
.93671 .92611 L
.92405 .92611 L
F
.87342 .90236 m
.88608 .90236 L
.88608 .92611 L
.87342 .92611 L
F
.91139 .90236 m
.92405 .90236 L
.92405 .92611 L
.91139 .92611 L
F
.89873 .90236 m
.91139 .90236 L
.91139 .92611 L
.89873 .92611 L
F
.70886 .11873 m
.72152 .11873 L
.72152 .14248 L
.70886 .14248 L
F
.43038 .14248 m
.44304 .14248 L
.44304 .16622 L
.43038 .16622 L
F
.6962 .11873 m
.70886 .11873 L
.70886 .14248 L
.6962 .14248 L
F
.46835 .14248 m
.48101 .14248 L
.48101 .16622 L
.46835 .16622 L
F
.41772 .14248 m
.43038 .14248 L
.43038 .16622 L
.41772 .16622 L
F
.68354 .11873 m
.6962 .11873 L
.6962 .14248 L
.68354 .14248 L
F
.65823 .11873 m
.67089 .11873 L
.67089 .14248 L
.65823 .14248 L
F
.77215 .11873 m
.78481 .11873 L
.78481 .14248 L
.77215 .14248 L
F
.72152 .11873 m
.73418 .11873 L
.73418 .14248 L
.72152 .14248 L
F
.67089 .11873 m
.68354 .11873 L
.68354 .14248 L
.67089 .14248 L
F
.40506 .14248 m
.41772 .14248 L
.41772 .16622 L
.40506 .16622 L
F
.37975 .14248 m
.39241 .14248 L
.39241 .16622 L
.37975 .16622 L
F
.35443 .14248 m
.36709 .14248 L
.36709 .16622 L
.35443 .16622 L
F
.56962 .14248 m
.58228 .14248 L
.58228 .16622 L
.56962 .16622 L
F
.39241 .14248 m
.40506 .14248 L
.40506 .16622 L
.39241 .16622 L
F
.4557 .14248 m
.46835 .14248 L
.46835 .16622 L
.4557 .16622 L
F
.44304 .14248 m
.4557 .14248 L
.4557 .16622 L
.44304 .16622 L
F
.34177 .14248 m
.35443 .14248 L
.35443 .16622 L
.34177 .16622 L
F
.36709 .14248 m
.37975 .14248 L
.37975 .16622 L
.36709 .16622 L
F
.64557 .11873 m
.65823 .11873 L
.65823 .14248 L
.64557 .14248 L
F
.62025 .11873 m
.63291 .11873 L
.63291 .14248 L
.62025 .14248 L
F
.59494 .11873 m
.60759 .11873 L
.60759 .14248 L
.59494 .14248 L
F
.5443 .11873 m
.55696 .11873 L
.55696 .14248 L
.5443 .14248 L
F
.63291 .11873 m
.64557 .11873 L
.64557 .14248 L
.63291 .14248 L
F
.96203 .11873 m
.97468 .11873 L
.97468 .14248 L
.96203 .14248 L
F
.98734 .11873 m
1 .11873 L
1 .14248 L
.98734 .14248 L
F
.60759 .11873 m
.62025 .11873 L
.62025 .14248 L
.60759 .14248 L
F
.97468 .11873 m
.98734 .11873 L
.98734 .14248 L
.97468 .14248 L
F
.74684 .11873 m
.75949 .11873 L
.75949 .14248 L
.74684 .14248 L
F
.73418 .11873 m
.74684 .11873 L
.74684 .14248 L
.73418 .14248 L
F
.56962 .11873 m
.58228 .11873 L
.58228 .14248 L
.56962 .14248 L
F
.75949 .11873 m
.77215 .11873 L
.77215 .14248 L
.75949 .14248 L
F
.78481 .11873 m
.79747 .11873 L
.79747 .14248 L
.78481 .14248 L
F
.53165 .11873 m
.5443 .11873 L
.5443 .14248 L
.53165 .14248 L
F
.51899 .11873 m
.53165 .11873 L
.53165 .14248 L
.51899 .14248 L
F
.55696 .11873 m
.56962 .11873 L
.56962 .14248 L
.55696 .14248 L
F
.58228 .11873 m
.59494 .11873 L
.59494 .14248 L
.58228 .14248 L
F
.20253 .33245 m
.21519 .33245 L
.21519 .3562 L
.20253 .3562 L
F
.17722 .33245 m
.18987 .33245 L
.18987 .3562 L
.17722 .3562 L
F
.1519 .33245 m
.16456 .33245 L
.16456 .3562 L
.1519 .3562 L
F
.36709 .33245 m
.37975 .33245 L
.37975 .3562 L
.36709 .3562 L
F
.18987 .33245 m
.20253 .33245 L
.20253 .3562 L
.18987 .3562 L
F
.24051 .33245 m
.25316 .33245 L
.25316 .3562 L
.24051 .3562 L
F
.25316 .33245 m
.26582 .33245 L
.26582 .3562 L
.25316 .3562 L
F
.26582 .33245 m
.27848 .33245 L
.27848 .3562 L
.26582 .3562 L
F
.13924 .33245 m
.1519 .33245 L
.1519 .3562 L
.13924 .3562 L
F
.16456 .33245 m
.17722 .33245 L
.17722 .3562 L
.16456 .3562 L
F
.29114 .33245 m
.3038 .33245 L
.3038 .3562 L
.29114 .3562 L
F
.34177 .33245 m
.35443 .33245 L
.35443 .3562 L
.34177 .3562 L
F
.27848 .33245 m
.29114 .33245 L
.29114 .3562 L
.27848 .3562 L
F
.32911 .33245 m
.34177 .33245 L
.34177 .3562 L
.32911 .3562 L
F
.31646 .33245 m
.32911 .33245 L
.32911 .3562 L
.31646 .3562 L
F
.40506 .33245 m
.41772 .33245 L
.41772 .3562 L
.40506 .3562 L
F
.37975 .33245 m
.39241 .33245 L
.39241 .3562 L
.37975 .3562 L
F
.35443 .33245 m
.36709 .33245 L
.36709 .3562 L
.35443 .3562 L
F
.3038 .33245 m
.31646 .33245 L
.31646 .3562 L
.3038 .3562 L
F
.39241 .33245 m
.40506 .33245 L
.40506 .3562 L
.39241 .3562 L
F
.49367 .3087 m
.50633 .3087 L
.50633 .33245 L
.49367 .33245 L
F
.48101 .3087 m
.49367 .3087 L
.49367 .33245 L
.48101 .33245 L
F
.31646 .3087 m
.32911 .3087 L
.32911 .33245 L
.31646 .33245 L
F
.50633 .3087 m
.51899 .3087 L
.51899 .33245 L
.50633 .33245 L
F
.53165 .3087 m
.5443 .3087 L
.5443 .33245 L
.53165 .33245 L
F
.26582 .3087 m
.27848 .3087 L
.27848 .33245 L
.26582 .33245 L
F
.27848 .3087 m
.29114 .3087 L
.29114 .33245 L
.27848 .33245 L
F
.29114 .3087 m
.3038 .3087 L
.3038 .33245 L
.29114 .33245 L
F
.3038 .3087 m
.31646 .3087 L
.31646 .33245 L
.3038 .33245 L
F
.32911 .3087 m
.34177 .3087 L
.34177 .33245 L
.32911 .33245 L
F
.44304 .3087 m
.4557 .3087 L
.4557 .33245 L
.44304 .33245 L
F
.4557 .3087 m
.46835 .3087 L
.46835 .33245 L
.4557 .33245 L
F
.46835 .3087 m
.48101 .3087 L
.48101 .33245 L
.46835 .33245 L
F
.21519 .33245 m
.22785 .33245 L
.22785 .3562 L
.21519 .3562 L
F
.22785 .33245 m
.24051 .33245 L
.24051 .3562 L
.22785 .3562 L
F
.43038 .3087 m
.44304 .3087 L
.44304 .33245 L
.43038 .33245 L
F
.51899 .3087 m
.53165 .3087 L
.53165 .33245 L
.51899 .33245 L
F
.41772 .3087 m
.43038 .3087 L
.43038 .33245 L
.41772 .33245 L
F
.40506 .3087 m
.41772 .3087 L
.41772 .33245 L
.40506 .33245 L
F
.16456 .40369 m
.17722 .40369 L
.17722 .42744 L
.16456 .42744 L
F
.17722 .40369 m
.18987 .40369 L
.18987 .42744 L
.17722 .42744 L
F
.18987 .40369 m
.20253 .40369 L
.20253 .42744 L
.18987 .42744 L
F
.07595 .40369 m
.08861 .40369 L
.08861 .42744 L
.07595 .42744 L
F
.08861 .40369 m
.10127 .40369 L
.10127 .42744 L
.08861 .42744 L
F
.03797 .33245 m
.05063 .33245 L
.05063 .3562 L
.03797 .3562 L
F
.05063 .33245 m
.06329 .33245 L
.06329 .3562 L
.05063 .3562 L
F
.06329 .33245 m
.07595 .33245 L
.07595 .3562 L
.06329 .3562 L
F
.13924 .40369 m
.1519 .40369 L
.1519 .42744 L
.13924 .42744 L
F
.1519 .40369 m
.16456 .40369 L
.16456 .42744 L
.1519 .42744 L
F
.3038 .40369 m
.31646 .40369 L
.31646 .42744 L
.3038 .42744 L
F
.31646 .40369 m
.32911 .40369 L
.32911 .42744 L
.31646 .42744 L
F
.32911 .40369 m
.34177 .40369 L
.34177 .42744 L
.32911 .42744 L
F
.20253 .40369 m
.21519 .40369 L
.21519 .42744 L
.20253 .42744 L
F
.22785 .40369 m
.24051 .40369 L
.24051 .42744 L
.22785 .42744 L
F
.10127 .40369 m
.11392 .40369 L
.11392 .42744 L
.10127 .42744 L
F
.11392 .40369 m
.12658 .40369 L
.12658 .42744 L
.11392 .42744 L
F
.12658 .40369 m
.13924 .40369 L
.13924 .42744 L
.12658 .42744 L
F
.27848 .40369 m
.29114 .40369 L
.29114 .42744 L
.27848 .42744 L
F
.29114 .40369 m
.3038 .40369 L
.3038 .42744 L
.29114 .42744 L
F
.86076 .3087 m
.87342 .3087 L
.87342 .33245 L
.86076 .33245 L
F
.87342 .3087 m
.88608 .3087 L
.88608 .33245 L
.87342 .33245 L
F
.88608 .3087 m
.89873 .3087 L
.89873 .33245 L
.88608 .33245 L
F
.89873 .3087 m
.91139 .3087 L
.91139 .33245 L
.89873 .33245 L
F
.92405 .3087 m
.93671 .3087 L
.93671 .33245 L
.92405 .33245 L
F
.93671 .3087 m
.94937 .3087 L
.94937 .33245 L
.93671 .33245 L
F
.98734 .3087 m
1 .3087 L
1 .33245 L
.98734 .33245 L
F
.94937 .3087 m
.96203 .3087 L
.96203 .33245 L
.94937 .33245 L
F
.97468 .3087 m
.98734 .3087 L
.98734 .33245 L
.97468 .33245 L
F
.96203 .3087 m
.97468 .3087 L
.97468 .33245 L
.96203 .33245 L
F
.11392 .33245 m
.12658 .33245 L
.12658 .3562 L
.11392 .3562 L
F
.02532 .33245 m
.03797 .33245 L
.03797 .3562 L
.02532 .3562 L
F
.10127 .33245 m
.11392 .33245 L
.11392 .3562 L
.10127 .3562 L
F
.01266 .33245 m
.02532 .33245 L
.02532 .3562 L
.01266 .3562 L
F
0 .33245 m
.01266 .33245 L
.01266 .3562 L
0 .3562 L
F
.08861 .33245 m
.10127 .33245 L
.10127 .3562 L
.08861 .3562 L
F
.91139 .3087 m
.92405 .3087 L
.92405 .33245 L
.91139 .33245 L
F
.12658 .33245 m
.13924 .33245 L
.13924 .3562 L
.12658 .3562 L
F
.07595 .33245 m
.08861 .33245 L
.08861 .3562 L
.07595 .3562 L
F
.56962 .3562 m
.58228 .3562 L
.58228 .37994 L
.56962 .37994 L
F
.48101 .3562 m
.49367 .3562 L
.49367 .37994 L
.48101 .37994 L
F
.55696 .3562 m
.56962 .3562 L
.56962 .37994 L
.55696 .37994 L
F
.46835 .3562 m
.48101 .3562 L
.48101 .37994 L
.46835 .37994 L
F
.4557 .3562 m
.46835 .3562 L
.46835 .37994 L
.4557 .37994 L
F
.36709 .3562 m
.37975 .3562 L
.37975 .37994 L
.36709 .37994 L
F
.5443 .3562 m
.55696 .3562 L
.55696 .37994 L
.5443 .37994 L
F
.35443 .3562 m
.36709 .3562 L
.36709 .37994 L
.35443 .37994 L
F
.58228 .3562 m
.59494 .3562 L
.59494 .37994 L
.58228 .37994 L
F
.53165 .3562 m
.5443 .3562 L
.5443 .37994 L
.53165 .37994 L
F
.13924 .3562 m
.1519 .3562 L
.1519 .37994 L
.13924 .37994 L
F
.1519 .3562 m
.16456 .3562 L
.16456 .37994 L
.1519 .37994 L
F
.16456 .3562 m
.17722 .3562 L
.17722 .37994 L
.16456 .37994 L
F
.03797 .3562 m
.05063 .3562 L
.05063 .37994 L
.03797 .37994 L
F
.06329 .3562 m
.07595 .3562 L
.07595 .37994 L
.06329 .37994 L
F
.49367 .3562 m
.50633 .3562 L
.50633 .37994 L
.49367 .37994 L
F
.50633 .3562 m
.51899 .3562 L
.51899 .37994 L
.50633 .37994 L
F
.51899 .3562 m
.53165 .3562 L
.53165 .37994 L
.51899 .37994 L
F
.11392 .3562 m
.12658 .3562 L
.12658 .37994 L
.11392 .37994 L
F
.12658 .3562 m
.13924 .3562 L
.13924 .37994 L
.12658 .37994 L
F
.56962 .33245 m
.58228 .33245 L
.58228 .3562 L
.56962 .3562 L
F
.62025 .33245 m
.63291 .33245 L
.63291 .3562 L
.62025 .3562 L
F
.55696 .33245 m
.56962 .33245 L
.56962 .3562 L
.55696 .3562 L
F
.60759 .33245 m
.62025 .33245 L
.62025 .3562 L
.60759 .3562 L
F
.59494 .33245 m
.60759 .33245 L
.60759 .3562 L
.59494 .3562 L
F
.68354 .33245 m
.6962 .33245 L
.6962 .3562 L
.68354 .3562 L
F
.65823 .33245 m
.67089 .33245 L
.67089 .3562 L
.65823 .3562 L
F
.63291 .33245 m
.64557 .33245 L
.64557 .3562 L
.63291 .3562 L
F
.58228 .33245 m
.59494 .33245 L
.59494 .3562 L
.58228 .3562 L
F
.67089 .33245 m
.68354 .33245 L
.68354 .3562 L
.67089 .3562 L
F
.34177 .3562 m
.35443 .3562 L
.35443 .37994 L
.34177 .37994 L
F
.31646 .3562 m
.32911 .3562 L
.32911 .37994 L
.31646 .37994 L
F
.43038 .3562 m
.44304 .3562 L
.44304 .37994 L
.43038 .37994 L
F
.37975 .3562 m
.39241 .3562 L
.39241 .37994 L
.37975 .37994 L
F
.32911 .3562 m
.34177 .3562 L
.34177 .37994 L
.32911 .37994 L
F
.39241 .3562 m
.40506 .3562 L
.40506 .37994 L
.39241 .37994 L
F
.40506 .3562 m
.41772 .3562 L
.41772 .37994 L
.40506 .37994 L
F
.41772 .3562 m
.43038 .3562 L
.43038 .37994 L
.41772 .37994 L
F
.44304 .3562 m
.4557 .3562 L
.4557 .37994 L
.44304 .37994 L
F
.81013 .3087 m
.82278 .3087 L
.82278 .33245 L
.81013 .33245 L
F
.8481 .3087 m
.86076 .3087 L
.86076 .33245 L
.8481 .33245 L
F
.58228 .3087 m
.59494 .3087 L
.59494 .33245 L
.58228 .33245 L
F
.83544 .3087 m
.8481 .3087 L
.8481 .33245 L
.83544 .33245 L
F
.82278 .3087 m
.83544 .3087 L
.83544 .33245 L
.82278 .33245 L
F
.55696 .3087 m
.56962 .3087 L
.56962 .33245 L
.55696 .33245 L
F
.5443 .3087 m
.55696 .3087 L
.55696 .33245 L
.5443 .33245 L
F
.64557 .3087 m
.65823 .3087 L
.65823 .33245 L
.64557 .33245 L
F
.56962 .3087 m
.58228 .3087 L
.58228 .33245 L
.56962 .33245 L
F
.59494 .3087 m
.60759 .3087 L
.60759 .33245 L
.59494 .33245 L
F
.34177 .3087 m
.35443 .3087 L
.35443 .33245 L
.34177 .33245 L
F
.39241 .3087 m
.40506 .3087 L
.40506 .33245 L
.39241 .33245 L
F
.35443 .3087 m
.36709 .3087 L
.36709 .33245 L
.35443 .33245 L
F
.37975 .3087 m
.39241 .3087 L
.39241 .33245 L
.37975 .33245 L
F
.36709 .3087 m
.37975 .3087 L
.37975 .33245 L
.36709 .33245 L
F
.68354 .3087 m
.6962 .3087 L
.6962 .33245 L
.68354 .33245 L
F
.79114 .3087 m
.79747 .3087 L
.79747 .32058 L
.79114 .32058 L
F
.67089 .3087 m
.68354 .3087 L
.68354 .33245 L
.67089 .33245 L
F
.77848 .3087 m
.78481 .3087 L
.78481 .32058 L
.77848 .32058 L
F
.71519 .3087 m
.72152 .3087 L
.72152 .32058 L
.71519 .32058 L
F
.3038 .3562 m
.31646 .3562 L
.31646 .37994 L
.3038 .37994 L
F
.27848 .3562 m
.29114 .3562 L
.29114 .37994 L
.27848 .37994 L
F
.25316 .3562 m
.26582 .3562 L
.26582 .37994 L
.25316 .37994 L
F
.20253 .3562 m
.21519 .3562 L
.21519 .37994 L
.20253 .37994 L
F
.29114 .3562 m
.3038 .3562 L
.3038 .37994 L
.29114 .37994 L
F
.10127 .3562 m
.11392 .3562 L
.11392 .37994 L
.10127 .37994 L
F
.07595 .3562 m
.08861 .3562 L
.08861 .37994 L
.07595 .37994 L
F
.05063 .3562 m
.06329 .3562 L
.06329 .37994 L
.05063 .37994 L
F
.26582 .3562 m
.27848 .3562 L
.27848 .37994 L
.26582 .37994 L
F
.08861 .3562 m
.10127 .3562 L
.10127 .37994 L
.08861 .37994 L
F
.62025 .3087 m
.63291 .3087 L
.63291 .33245 L
.62025 .33245 L
F
.60759 .3087 m
.62025 .3087 L
.62025 .33245 L
.60759 .33245 L
F
.22785 .3562 m
.24051 .3562 L
.24051 .37994 L
.22785 .37994 L
F
.63291 .3087 m
.64557 .3087 L
.64557 .33245 L
.63291 .33245 L
F
.65823 .3087 m
.67089 .3087 L
.67089 .33245 L
.65823 .33245 L
F
.18987 .3562 m
.20253 .3562 L
.20253 .37994 L
.18987 .37994 L
F
.17722 .3562 m
.18987 .3562 L
.18987 .37994 L
.17722 .37994 L
F
.21519 .3562 m
.22785 .3562 L
.22785 .37994 L
.21519 .37994 L
F
.24051 .3562 m
.25316 .3562 L
.25316 .37994 L
.24051 .37994 L
F
.93671 .3562 m
.94937 .3562 L
.94937 .37994 L
.93671 .37994 L
F
.91139 .3562 m
.92405 .3562 L
.92405 .37994 L
.91139 .37994 L
F
.88608 .3562 m
.89873 .3562 L
.89873 .37994 L
.88608 .37994 L
F
.83544 .3562 m
.8481 .3562 L
.8481 .37994 L
.83544 .37994 L
F
.92405 .3562 m
.93671 .3562 L
.93671 .37994 L
.92405 .37994 L
F
.65823 .3562 m
.67089 .3562 L
.67089 .37994 L
.65823 .37994 L
F
.63291 .3562 m
.64557 .3562 L
.64557 .37994 L
.63291 .37994 L
F
.60759 .3562 m
.62025 .3562 L
.62025 .37994 L
.60759 .37994 L
F
.89873 .3562 m
.91139 .3562 L
.91139 .37994 L
.89873 .37994 L
F
.64557 .3562 m
.65823 .3562 L
.65823 .37994 L
.64557 .37994 L
F
.58228 .37994 m
.59494 .37994 L
.59494 .40369 L
.58228 .40369 L
F
.63291 .37994 m
.64557 .37994 L
.64557 .40369 L
.63291 .40369 L
F
.59494 .37994 m
.60759 .37994 L
.60759 .40369 L
.59494 .40369 L
F
.62025 .37994 m
.63291 .37994 L
.63291 .40369 L
.62025 .40369 L
F
.60759 .37994 m
.62025 .37994 L
.62025 .40369 L
.60759 .40369 L
F
.82278 .3562 m
.83544 .3562 L
.83544 .37994 L
.82278 .37994 L
F
.87342 .3562 m
.88608 .3562 L
.88608 .37994 L
.87342 .37994 L
F
.81013 .3562 m
.82278 .3562 L
.82278 .37994 L
.81013 .37994 L
F
.86076 .3562 m
.87342 .3562 L
.87342 .37994 L
.86076 .37994 L
F
.8481 .3562 m
.86076 .3562 L
.86076 .37994 L
.8481 .37994 L
F
.18987 .37994 m
.20253 .37994 L
.20253 .40369 L
.18987 .40369 L
F
.20253 .37994 m
.21519 .37994 L
.21519 .40369 L
.20253 .40369 L
F
.21519 .37994 m
.22785 .37994 L
.22785 .40369 L
.21519 .40369 L
F
.08861 .37994 m
.10127 .37994 L
.10127 .40369 L
.08861 .40369 L
F
.11392 .37994 m
.12658 .37994 L
.12658 .40369 L
.11392 .40369 L
F
.98734 .3562 m
1 .3562 L
1 .37994 L
.98734 .37994 L
F
0 .37994 m
.01266 .37994 L
.01266 .40369 L
0 .40369 L
F
.01266 .37994 m
.02532 .37994 L
.02532 .40369 L
.01266 .40369 L
F
.16456 .37994 m
.17722 .37994 L
.17722 .40369 L
.16456 .40369 L
F
.17722 .37994 m
.18987 .37994 L
.18987 .40369 L
.17722 .40369 L
F
.67089 .3562 m
.68354 .3562 L
.68354 .37994 L
.67089 .37994 L
F
.79747 .3562 m
.81013 .3562 L
.81013 .37994 L
.79747 .37994 L
F
.68354 .3562 m
.6962 .3562 L
.6962 .37994 L
.68354 .37994 L
F
.59494 .3562 m
.60759 .3562 L
.60759 .37994 L
.59494 .37994 L
F
.62025 .3562 m
.63291 .3562 L
.63291 .37994 L
.62025 .37994 L
F
.1519 .37994 m
.16456 .37994 L
.16456 .40369 L
.1519 .40369 L
F
.10127 .37994 m
.11392 .37994 L
.11392 .40369 L
.10127 .40369 L
F
.13924 .37994 m
.1519 .37994 L
.1519 .40369 L
.13924 .40369 L
F
.12658 .37994 m
.13924 .37994 L
.13924 .40369 L
.12658 .40369 L
F
.44304 .37994 m
.4557 .37994 L
.4557 .40369 L
.44304 .40369 L
F
.49367 .37994 m
.50633 .37994 L
.50633 .40369 L
.49367 .40369 L
F
.4557 .37994 m
.46835 .37994 L
.46835 .40369 L
.4557 .40369 L
F
.48101 .37994 m
.49367 .37994 L
.49367 .40369 L
.48101 .40369 L
F
.46835 .37994 m
.48101 .37994 L
.48101 .40369 L
.46835 .40369 L
F
.24051 .37994 m
.25316 .37994 L
.25316 .40369 L
.24051 .40369 L
F
.29114 .37994 m
.3038 .37994 L
.3038 .40369 L
.29114 .40369 L
F
.22785 .37994 m
.24051 .37994 L
.24051 .40369 L
.22785 .40369 L
F
.27848 .37994 m
.29114 .37994 L
.29114 .40369 L
.27848 .40369 L
F
.26582 .37994 m
.27848 .37994 L
.27848 .40369 L
.26582 .40369 L
F
.3038 .23746 m
.31646 .23746 L
.31646 .26121 L
.3038 .26121 L
F
.29114 .23746 m
.3038 .23746 L
.3038 .26121 L
.29114 .26121 L
F
.41772 .37994 m
.43038 .37994 L
.43038 .40369 L
.41772 .40369 L
F
.31646 .23746 m
.32911 .23746 L
.32911 .26121 L
.31646 .26121 L
F
.34177 .23746 m
.35443 .23746 L
.35443 .26121 L
.34177 .26121 L
F
.36709 .37994 m
.37975 .37994 L
.37975 .40369 L
.36709 .40369 L
F
.37975 .37994 m
.39241 .37994 L
.39241 .40369 L
.37975 .40369 L
F
.39241 .37994 m
.40506 .37994 L
.40506 .40369 L
.39241 .40369 L
F
.40506 .37994 m
.41772 .37994 L
.41772 .40369 L
.40506 .40369 L
F
.43038 .37994 m
.44304 .37994 L
.44304 .40369 L
.43038 .40369 L
F
.78481 .37994 m
.79747 .37994 L
.79747 .40369 L
.78481 .40369 L
F
.75949 .38202 m
.75949 .40369 L
.74684 .40369 L
.74684 .38254 L
F
.55696 .37994 m
.56962 .37994 L
.56962 .40369 L
.55696 .40369 L
F
.64557 .37994 m
.65823 .37994 L
.65823 .40369 L
.64557 .40369 L
F
.67089 .37994 m
.68354 .37994 L
.68354 .40369 L
.67089 .40369 L
F
.50633 .37994 m
.51899 .37994 L
.51899 .40369 L
.50633 .40369 L
F
.51899 .37994 m
.53165 .37994 L
.53165 .40369 L
.51899 .40369 L
F
.53165 .37994 m
.5443 .37994 L
.5443 .40369 L
.53165 .40369 L
F
.5443 .37994 m
.55696 .37994 L
.55696 .40369 L
.5443 .40369 L
F
.56962 .37994 m
.58228 .37994 L
.58228 .40369 L
.56962 .40369 L
F
.35443 .37994 m
.36709 .37994 L
.36709 .40369 L
.35443 .40369 L
F
.32911 .37994 m
.34177 .37994 L
.34177 .40369 L
.32911 .40369 L
F
.3038 .37994 m
.31646 .37994 L
.31646 .40369 L
.3038 .40369 L
F
.25316 .37994 m
.26582 .37994 L
.26582 .40369 L
.25316 .40369 L
F
.34177 .37994 m
.35443 .37994 L
.35443 .40369 L
.34177 .40369 L
F
.68354 .37994 m
.6962 .37994 L
.6962 .40369 L
.68354 .40369 L
F
.65823 .37994 m
.67089 .37994 L
.67089 .40369 L
.65823 .40369 L
F
.31646 .37994 m
.32911 .37994 L
.32911 .40369 L
.31646 .40369 L
F
.6962 .37994 m
.70886 .37994 L
.70886 .40369 L
.6962 .40369 L
F
.98734 .37994 m
1 .37994 L
1 .40369 L
.98734 .40369 L
F
.70886 .40369 m
.72152 .40369 L
.72152 .42744 L
.70886 .42744 L
F
.97468 .37994 m
.98734 .37994 L
.98734 .40369 L
.97468 .40369 L
F
.74684 .40369 m
.75949 .40369 L
.75949 .42744 L
.74684 .42744 L
F
.6962 .40369 m
.70886 .40369 L
.70886 .42744 L
.6962 .42744 L
F
.96203 .37994 m
.97468 .37994 L
.97468 .40369 L
.96203 .40369 L
F
.93671 .37994 m
.94937 .37994 L
.94937 .40369 L
.93671 .40369 L
F
.05063 .40369 m
.06329 .40369 L
.06329 .42744 L
.05063 .42744 L
F
0 .40369 m
.01266 .40369 L
.01266 .42744 L
0 .42744 L
F
.94937 .37994 m
.96203 .37994 L
.96203 .40369 L
.94937 .40369 L
F
.65823 .40369 m
.67089 .40369 L
.67089 .42744 L
.65823 .42744 L
F
.67089 .40369 m
.68354 .40369 L
.68354 .42744 L
.67089 .42744 L
F
.68354 .40369 m
.6962 .40369 L
.6962 .42744 L
.68354 .42744 L
F
.83544 .40369 m
.8481 .40369 L
.8481 .42744 L
.83544 .42744 L
F
.8481 .40369 m
.86076 .40369 L
.86076 .42744 L
.8481 .42744 L
F
.73418 .40369 m
.74684 .40369 L
.74684 .42744 L
.73418 .42744 L
F
.64557 .40369 m
.65823 .40369 L
.65823 .42744 L
.64557 .42744 L
F
.72152 .40369 m
.73418 .40369 L
.73418 .42744 L
.72152 .42744 L
F
.63291 .40369 m
.64557 .40369 L
.64557 .42744 L
.63291 .42744 L
F
.62025 .40369 m
.63291 .40369 L
.63291 .42744 L
.62025 .42744 L
F
.92405 .37994 m
.93671 .37994 L
.93671 .40369 L
.92405 .40369 L
F
.89873 .37994 m
.91139 .37994 L
.91139 .40369 L
.89873 .40369 L
F
.87342 .37994 m
.88608 .37994 L
.88608 .40369 L
.87342 .40369 L
F
.82278 .37994 m
.83544 .37994 L
.83544 .40369 L
.82278 .40369 L
F
.91139 .37994 m
.92405 .37994 L
.92405 .40369 L
.91139 .40369 L
F
.26582 .40369 m
.27848 .40369 L
.27848 .42744 L
.26582 .42744 L
F
.24051 .40369 m
.25316 .40369 L
.25316 .42744 L
.24051 .42744 L
F
.21519 .40369 m
.22785 .40369 L
.22785 .42744 L
.21519 .42744 L
F
.88608 .37994 m
.89873 .37994 L
.89873 .40369 L
.88608 .40369 L
F
.25316 .40369 m
.26582 .40369 L
.26582 .42744 L
.25316 .42744 L
F
.02532 .40369 m
.03797 .40369 L
.03797 .42744 L
.02532 .42744 L
F
.01266 .40369 m
.02532 .40369 L
.02532 .42744 L
.01266 .42744 L
F
.8481 .37994 m
.86076 .37994 L
.86076 .40369 L
.8481 .40369 L
F
.03797 .40369 m
.05063 .40369 L
.05063 .42744 L
.03797 .42744 L
F
.06329 .40369 m
.07595 .40369 L
.07595 .42744 L
.06329 .42744 L
F
.81013 .37994 m
.82278 .37994 L
.82278 .40369 L
.81013 .40369 L
F
.79747 .37994 m
.81013 .37994 L
.81013 .40369 L
.79747 .40369 L
F
.83544 .37994 m
.8481 .37994 L
.8481 .40369 L
.83544 .40369 L
F
.86076 .37994 m
.87342 .37994 L
.87342 .40369 L
.86076 .40369 L
F
.50633 .40369 m
.51899 .40369 L
.51899 .42744 L
.50633 .42744 L
F
.48101 .40369 m
.49367 .40369 L
.49367 .42744 L
.48101 .42744 L
F
.59494 .40369 m
.60759 .40369 L
.60759 .42744 L
.59494 .42744 L
F
.5443 .40369 m
.55696 .40369 L
.55696 .42744 L
.5443 .42744 L
F
.49367 .40369 m
.50633 .40369 L
.50633 .42744 L
.49367 .42744 L
F
.56962 .40369 m
.58228 .40369 L
.58228 .42744 L
.56962 .42744 L
F
.55696 .40369 m
.56962 .40369 L
.56962 .42744 L
.55696 .42744 L
F
.39241 .40369 m
.40506 .40369 L
.40506 .42744 L
.39241 .42744 L
F
.58228 .40369 m
.59494 .40369 L
.59494 .42744 L
.58228 .42744 L
F
.60759 .40369 m
.62025 .40369 L
.62025 .42744 L
.60759 .42744 L
F
.06329 .37994 m
.07595 .37994 L
.07595 .40369 L
.06329 .40369 L
F
.97468 .3562 m
.98734 .3562 L
.98734 .37994 L
.97468 .37994 L
F
.05063 .37994 m
.06329 .37994 L
.06329 .40369 L
.05063 .40369 L
F
.96203 .3562 m
.97468 .3562 L
.97468 .37994 L
.96203 .37994 L
F
.94937 .3562 m
.96203 .3562 L
.96203 .37994 L
.94937 .37994 L
F
.53165 .40369 m
.5443 .40369 L
.5443 .42744 L
.53165 .42744 L
F
.03797 .37994 m
.05063 .37994 L
.05063 .40369 L
.03797 .40369 L
F
.51899 .40369 m
.53165 .40369 L
.53165 .42744 L
.51899 .42744 L
F
.07595 .37994 m
.08861 .37994 L
.08861 .40369 L
.07595 .40369 L
F
.02532 .37994 m
.03797 .37994 L
.03797 .40369 L
.02532 .40369 L
F
.82278 .40369 m
.83544 .40369 L
.83544 .42744 L
.82278 .42744 L
F
.79747 .40369 m
.81013 .40369 L
.81013 .42744 L
.79747 .42744 L
F
.77215 .40369 m
.78481 .40369 L
.78481 .42744 L
.77215 .42744 L
F
.43038 .40369 m
.44304 .40369 L
.44304 .42744 L
.43038 .42744 L
F
.81013 .40369 m
.82278 .40369 L
.82278 .42744 L
.81013 .42744 L
F
.86076 .40369 m
.87342 .40369 L
.87342 .42744 L
.86076 .42744 L
F
.87342 .40369 m
.88608 .40369 L
.88608 .42744 L
.87342 .42744 L
F
.88608 .40369 m
.89873 .40369 L
.89873 .42744 L
.88608 .42744 L
F
.75949 .40369 m
.77215 .40369 L
.77215 .42744 L
.75949 .42744 L
F
.78481 .40369 m
.79747 .40369 L
.79747 .42744 L
.78481 .42744 L
F
.34177 .40369 m
.35443 .40369 L
.35443 .42744 L
.34177 .42744 L
F
.35443 .40369 m
.36709 .40369 L
.36709 .42744 L
.35443 .42744 L
F
.36709 .40369 m
.37975 .40369 L
.37975 .42744 L
.36709 .42744 L
F
.37975 .40369 m
.39241 .40369 L
.39241 .42744 L
.37975 .42744 L
F
.40506 .40369 m
.41772 .40369 L
.41772 .42744 L
.40506 .42744 L
F
.46835 .40369 m
.48101 .40369 L
.48101 .42744 L
.46835 .42744 L
F
.41772 .40369 m
.43038 .40369 L
.43038 .42744 L
.41772 .42744 L
F
.4557 .40369 m
.46835 .40369 L
.46835 .42744 L
.4557 .42744 L
F
.44304 .40369 m
.4557 .40369 L
.4557 .42744 L
.44304 .42744 L
F
.72152 0 m
.73418 0 L
.73418 .02375 L
.72152 .02375 L
F
.73418 0 m
.74684 0 L
.74684 .02375 L
.73418 .02375 L
F
.74684 0 m
.75949 0 L
.75949 .02375 L
.74684 .02375 L
F
.65823 .07124 m
.67089 .07124 L
.67089 .09499 L
.65823 .09499 L
F
.67089 .07124 m
.68354 .07124 L
.68354 .09499 L
.67089 .09499 L
F
.79747 0 m
.81013 0 L
.81013 .02375 L
.79747 .02375 L
F
.70886 0 m
.72152 0 L
.72152 .02375 L
.70886 .02375 L
F
.78481 0 m
.79747 0 L
.79747 .02375 L
.78481 .02375 L
F
.6962 0 m
.70886 0 L
.70886 .02375 L
.6962 .02375 L
F
.68354 0 m
.6962 0 L
.6962 .02375 L
.68354 .02375 L
F
.62025 .07124 m
.63291 .07124 L
.63291 .09499 L
.62025 .09499 L
F
.63291 .07124 m
.64557 .07124 L
.64557 .09499 L
.63291 .09499 L
F
.64557 .07124 m
.65823 .07124 L
.65823 .09499 L
.64557 .09499 L
F
.79747 .07124 m
.81013 .07124 L
.81013 .09499 L
.79747 .09499 L
F
.81013 .07124 m
.82278 .07124 L
.82278 .09499 L
.81013 .09499 L
F
.68354 .07124 m
.6962 .07124 L
.6962 .09499 L
.68354 .09499 L
F
.6962 .07124 m
.70886 .07124 L
.70886 .09499 L
.6962 .09499 L
F
.70886 .07124 m
.72152 .07124 L
.72152 .09499 L
.70886 .09499 L
F
.59494 .07124 m
.60759 .07124 L
.60759 .09499 L
.59494 .09499 L
F
.60759 .07124 m
.62025 .07124 L
.62025 .09499 L
.60759 .09499 L
F
.62025 0 m
.63291 0 L
.63291 .02375 L
.62025 .02375 L
F
.67089 0 m
.68354 0 L
.68354 .02375 L
.67089 .02375 L
F
.63291 0 m
.64557 0 L
.64557 .02375 L
.63291 .02375 L
F
.65823 0 m
.67089 0 L
.67089 .02375 L
.65823 .02375 L
F
.64557 0 m
.65823 0 L
.65823 .02375 L
.64557 .02375 L
F
.97468 0 m
.98734 0 L
.98734 .02375 L
.97468 .02375 L
F
.02532 .02375 m
.03797 .02375 L
.03797 .04749 L
.02532 .04749 L
F
.96203 0 m
.97468 0 L
.97468 .02375 L
.96203 .02375 L
F
.01266 .02375 m
.02532 .02375 L
.02532 .04749 L
.01266 .04749 L
F
0 .02375 m
.01266 .02375 L
.01266 .04749 L
0 .04749 L
F
.59494 0 m
.60759 0 L
.60759 .02375 L
.59494 .02375 L
F
.77215 0 m
.78481 0 L
.78481 .02375 L
.77215 .02375 L
F
.58228 0 m
.59494 0 L
.59494 .02375 L
.58228 .02375 L
F
.81013 0 m
.82278 0 L
.82278 .02375 L
.81013 .02375 L
F
.75949 0 m
.77215 0 L
.77215 .02375 L
.75949 .02375 L
F
.5443 0 m
.55696 0 L
.55696 .02375 L
.5443 .02375 L
F
.56962 0 m
.58228 0 L
.58228 .02375 L
.56962 .02375 L
F
.60759 0 m
.62025 0 L
.62025 .02375 L
.60759 .02375 L
F
.55696 0 m
.56962 0 L
.56962 .02375 L
.55696 .02375 L
F
.48101 .07124 m
.49367 .07124 L
.49367 .09499 L
.48101 .09499 L
F
.4557 .07124 m
.46835 .07124 L
.46835 .09499 L
.4557 .09499 L
F
.56962 .07124 m
.58228 .07124 L
.58228 .09499 L
.56962 .09499 L
F
.51899 .07124 m
.53165 .07124 L
.53165 .09499 L
.51899 .09499 L
F
.46835 .07124 m
.48101 .07124 L
.48101 .09499 L
.46835 .09499 L
F
.5443 .07124 m
.55696 .07124 L
.55696 .09499 L
.5443 .09499 L
F
.53165 .07124 m
.5443 .07124 L
.5443 .09499 L
.53165 .09499 L
F
.36709 .07124 m
.37975 .07124 L
.37975 .09499 L
.36709 .09499 L
F
.55696 .07124 m
.56962 .07124 L
.56962 .09499 L
.55696 .09499 L
F
.58228 .07124 m
.59494 .07124 L
.59494 .09499 L
.58228 .09499 L
F
.25316 .09499 m
.26582 .09499 L
.26582 .11873 L
.25316 .11873 L
F
.16456 .09499 m
.17722 .09499 L
.17722 .11873 L
.16456 .11873 L
F
.24051 .09499 m
.25316 .09499 L
.25316 .11873 L
.24051 .11873 L
F
.1519 .09499 m
.16456 .09499 L
.16456 .11873 L
.1519 .11873 L
F
.13924 .09499 m
.1519 .09499 L
.1519 .11873 L
.13924 .11873 L
F
.50633 .07124 m
.51899 .07124 L
.51899 .09499 L
.50633 .09499 L
F
.22785 .09499 m
.24051 .09499 L
.24051 .11873 L
.22785 .11873 L
F
.49367 .07124 m
.50633 .07124 L
.50633 .09499 L
.49367 .09499 L
F
.26582 .09499 m
.27848 .09499 L
.27848 .11873 L
.26582 .11873 L
F
.21519 .09499 m
.22785 .09499 L
.22785 .11873 L
.21519 .11873 L
F
.78481 .07124 m
.79747 .07124 L
.79747 .09499 L
.78481 .09499 L
F
.75949 .07124 m
.77215 .07124 L
.77215 .09499 L
.75949 .09499 L
F
.73418 .07124 m
.74684 .07124 L
.74684 .09499 L
.73418 .09499 L
F
.40506 .07124 m
.41772 .07124 L
.41772 .09499 L
.40506 .09499 L
F
.77215 .07124 m
.78481 .07124 L
.78481 .09499 L
.77215 .09499 L
F
.82278 .07124 m
.83544 .07124 L
.83544 .09499 L
.82278 .09499 L
F
.83544 .07124 m
.8481 .07124 L
.8481 .09499 L
.83544 .09499 L
F
.8481 .07124 m
.86076 .07124 L
.86076 .09499 L
.8481 .09499 L
F
.72152 .07124 m
.73418 .07124 L
.73418 .09499 L
.72152 .09499 L
F
.74684 .07124 m
.75949 .07124 L
.75949 .09499 L
.74684 .09499 L
F
.31646 .07124 m
.32911 .07124 L
.32911 .09499 L
.31646 .09499 L
F
.32911 .07124 m
.34177 .07124 L
.34177 .09499 L
.32911 .09499 L
F
.34177 .07124 m
.35443 .07124 L
.35443 .09499 L
.34177 .09499 L
F
.35443 .07124 m
.36709 .07124 L
.36709 .09499 L
.35443 .09499 L
F
.37975 .07124 m
.39241 .07124 L
.39241 .09499 L
.37975 .09499 L
F
.44304 .07124 m
.4557 .07124 L
.4557 .09499 L
.44304 .09499 L
F
.39241 .07124 m
.40506 .07124 L
.40506 .09499 L
.39241 .09499 L
F
.43038 .07124 m
.44304 .07124 L
.44304 .09499 L
.43038 .09499 L
F
.41772 .07124 m
.43038 .07124 L
.43038 .09499 L
.41772 .09499 L
F
.32911 0 m
.34177 0 L
.34177 .02375 L
.32911 .02375 L
F
.3038 0 m
.31646 0 L
.31646 .02375 L
.3038 .02375 L
F
.27848 0 m
.29114 0 L
.29114 .02375 L
.27848 .02375 L
F
.49367 0 m
.50633 0 L
.50633 .02375 L
.49367 .02375 L
F
.31646 0 m
.32911 0 L
.32911 .02375 L
.31646 .02375 L
F
.39241 0 m
.40506 0 L
.40506 .02375 L
.39241 .02375 L
F
.36709 0 m
.37975 0 L
.37975 .02375 L
.36709 .02375 L
F
.34177 0 m
.35443 0 L
.35443 .02375 L
.34177 .02375 L
F
.29114 0 m
.3038 0 L
.3038 .02375 L
.29114 .02375 L
F
.37975 0 m
.39241 0 L
.39241 .02375 L
.37975 .02375 L
F
.41772 0 m
.43038 0 L
.43038 .02375 L
.41772 .02375 L
F
.46835 0 m
.48101 0 L
.48101 .02375 L
.46835 .02375 L
F
.40506 0 m
.41772 0 L
.41772 .02375 L
.40506 .02375 L
F
.4557 0 m
.46835 0 L
.46835 .02375 L
.4557 .02375 L
F
.44304 0 m
.4557 0 L
.4557 .02375 L
.44304 .02375 L
F
.53165 0 m
.5443 0 L
.5443 .02375 L
.53165 .02375 L
F
.50633 0 m
.51899 0 L
.51899 .02375 L
.50633 .02375 L
F
.48101 0 m
.49367 0 L
.49367 .02375 L
.48101 .02375 L
F
.43038 0 m
.44304 0 L
.44304 .02375 L
.43038 .02375 L
F
.51899 0 m
.53165 0 L
.53165 .02375 L
.51899 .02375 L
F
.6962 .02375 m
.70886 .02375 L
.70886 .04749 L
.6962 .04749 L
F
.87342 .02375 m
.88608 .02375 L
.88608 .04749 L
.87342 .04749 L
F
.68354 .02375 m
.6962 .02375 L
.6962 .04749 L
.68354 .04749 L
F
.91139 .02375 m
.92405 .02375 L
.92405 .04749 L
.91139 .04749 L
F
.86076 .02375 m
.87342 .02375 L
.87342 .04749 L
.86076 .04749 L
F
.67089 .02375 m
.68354 .02375 L
.68354 .04749 L
.67089 .04749 L
F
.64557 .02375 m
.65823 .02375 L
.65823 .04749 L
.64557 .04749 L
F
.75949 .02375 m
.77215 .02375 L
.77215 .04749 L
.75949 .04749 L
F
.70886 .02375 m
.72152 .02375 L
.72152 .04749 L
.70886 .04749 L
F
.65823 .02375 m
.67089 .02375 L
.67089 .04749 L
.65823 .04749 L
F
.8481 .02375 m
.86076 .02375 L
.86076 .04749 L
.8481 .04749 L
F
.82278 .02375 m
.83544 .02375 L
.83544 .04749 L
.82278 .04749 L
F
.79747 .02375 m
.81013 .02375 L
.81013 .04749 L
.79747 .04749 L
F
.35443 0 m
.36709 0 L
.36709 .02375 L
.35443 .02375 L
F
.83544 .02375 m
.8481 .02375 L
.8481 .04749 L
.83544 .04749 L
F
.89873 .02375 m
.91139 .02375 L
.91139 .04749 L
.89873 .04749 L
F
.88608 .02375 m
.89873 .02375 L
.89873 .04749 L
.88608 .04749 L
F
.78481 .02375 m
.79747 .02375 L
.79747 .04749 L
.78481 .04749 L
F
.81013 .02375 m
.82278 .02375 L
.82278 .04749 L
.81013 .04749 L
F
.92405 0 m
.93671 0 L
.93671 .02375 L
.92405 .02375 L
F
.93671 0 m
.94937 0 L
.94937 .02375 L
.93671 .02375 L
F
.94937 0 m
.96203 0 L
.96203 .02375 L
.94937 .02375 L
F
.82278 0 m
.83544 0 L
.83544 .02375 L
.82278 .02375 L
F
.8481 0 m
.86076 0 L
.86076 .02375 L
.8481 .02375 L
F
.17722 0 m
.18987 0 L
.18987 .02375 L
.17722 .02375 L
F
.18987 0 m
.20253 0 L
.20253 .02375 L
.18987 .02375 L
F
.20253 0 m
.21519 0 L
.21519 .02375 L
.20253 .02375 L
F
.89873 0 m
.91139 0 L
.91139 .02375 L
.89873 .02375 L
F
.91139 0 m
.92405 0 L
.92405 .02375 L
.91139 .02375 L
F
.08861 .02375 m
.10127 .02375 L
.10127 .04749 L
.08861 .04749 L
F
.06329 .02375 m
.07595 .02375 L
.07595 .04749 L
.06329 .04749 L
F
.03797 .02375 m
.05063 .02375 L
.05063 .04749 L
.03797 .04749 L
F
.98734 0 m
1 0 L
1 .02375 L
.98734 .02375 L
F
.07595 .02375 m
.08861 .02375 L
.08861 .04749 L
.07595 .04749 L
F
.88608 0 m
.89873 0 L
.89873 .02375 L
.88608 .02375 L
F
.86076 0 m
.87342 0 L
.87342 .02375 L
.86076 .02375 L
F
.83544 0 m
.8481 0 L
.8481 .02375 L
.83544 .02375 L
F
.05063 .02375 m
.06329 .02375 L
.06329 .04749 L
.05063 .04749 L
F
.87342 0 m
.88608 0 L
.88608 .02375 L
.87342 .02375 L
F
0 0 m
.01266 0 L
.01266 .02375 L
0 .02375 L
F
.01266 0 m
.02532 0 L
.02532 .02375 L
.01266 .02375 L
F
.02532 0 m
.03797 0 L
.03797 .02375 L
.02532 .02375 L
F
.03797 0 m
.05063 0 L
.05063 .02375 L
.03797 .02375 L
F
.06329 0 m
.07595 0 L
.07595 .02375 L
.06329 .02375 L
F
.07595 0 m
.08861 0 L
.08861 .02375 L
.07595 .02375 L
F
.12658 0 m
.13924 0 L
.13924 .02375 L
.12658 .02375 L
F
.08861 0 m
.10127 0 L
.10127 .02375 L
.08861 .02375 L
F
.11392 0 m
.12658 0 L
.12658 .02375 L
.11392 .02375 L
F
.10127 0 m
.11392 0 L
.11392 .02375 L
.10127 .02375 L
F
.25316 0 m
.26582 0 L
.26582 .02375 L
.25316 .02375 L
F
.16456 0 m
.17722 0 L
.17722 .02375 L
.16456 .02375 L
F
.24051 0 m
.25316 0 L
.25316 .02375 L
.24051 .02375 L
F
.1519 0 m
.16456 0 L
.16456 .02375 L
.1519 .02375 L
F
.13924 0 m
.1519 0 L
.1519 .02375 L
.13924 .02375 L
F
.22785 0 m
.24051 0 L
.24051 .02375 L
.22785 .02375 L
F
.05063 0 m
.06329 0 L
.06329 .02375 L
.05063 .02375 L
F
.26582 0 m
.27848 0 L
.27848 .02375 L
.26582 .02375 L
F
.21519 0 m
.22785 0 L
.22785 .02375 L
.21519 .02375 L
F
.8481 .04749 m
.86076 .04749 L
.86076 .07124 L
.8481 .07124 L
F
.83544 .04749 m
.8481 .04749 L
.8481 .07124 L
.83544 .07124 L
F
.22785 .07124 m
.24051 .07124 L
.24051 .09499 L
.22785 .09499 L
F
.86076 .04749 m
.87342 .04749 L
.87342 .07124 L
.86076 .07124 L
F
.88608 .04749 m
.89873 .04749 L
.89873 .07124 L
.88608 .07124 L
F
.17722 .07124 m
.18987 .07124 L
.18987 .09499 L
.17722 .09499 L
F
.18987 .07124 m
.20253 .07124 L
.20253 .09499 L
.18987 .09499 L
F
.20253 .07124 m
.21519 .07124 L
.21519 .09499 L
.20253 .09499 L
F
.21519 .07124 m
.22785 .07124 L
.22785 .09499 L
.21519 .09499 L
F
.24051 .07124 m
.25316 .07124 L
.25316 .09499 L
.24051 .09499 L
F
.81013 .04749 m
.82278 .04749 L
.82278 .07124 L
.81013 .07124 L
F
.98734 .04749 m
1 .04749 L
1 .07124 L
.98734 .07124 L
F
.79747 .04749 m
.81013 .04749 L
.81013 .07124 L
.79747 .07124 L
F
.02532 .07124 m
.03797 .07124 L
.03797 .09499 L
.02532 .09499 L
F
.97468 .04749 m
.98734 .04749 L
.98734 .07124 L
.97468 .07124 L
F
.78481 .04749 m
.79747 .04749 L
.79747 .07124 L
.78481 .07124 L
F
.75949 .04749 m
.77215 .04749 L
.77215 .07124 L
.75949 .07124 L
F
.87342 .04749 m
.88608 .04749 L
.88608 .07124 L
.87342 .07124 L
F
.82278 .04749 m
.83544 .04749 L
.83544 .07124 L
.82278 .07124 L
F
.77215 .04749 m
.78481 .04749 L
.78481 .07124 L
.77215 .07124 L
F
.13924 .07124 m
.1519 .07124 L
.1519 .09499 L
.13924 .09499 L
F
.1519 .07124 m
.16456 .07124 L
.16456 .09499 L
.1519 .09499 L
F
.16456 .07124 m
.17722 .07124 L
.17722 .09499 L
.16456 .09499 L
F
.03797 .07124 m
.05063 .07124 L
.05063 .09499 L
.03797 .09499 L
F
.06329 .07124 m
.07595 .07124 L
.07595 .09499 L
.06329 .09499 L
F
.37975 .04749 m
.39241 .04749 L
.39241 .07124 L
.37975 .07124 L
F
.39241 .04749 m
.40506 .04749 L
.40506 .07124 L
.39241 .07124 L
F
.40506 .04749 m
.41772 .04749 L
.41772 .07124 L
.40506 .07124 L
F
.11392 .07124 m
.12658 .07124 L
.12658 .09499 L
.11392 .09499 L
F
.12658 .07124 m
.13924 .07124 L
.13924 .09499 L
.12658 .09499 L
F
.25316 .07124 m
.26582 .07124 L
.26582 .09499 L
.25316 .09499 L
F
.3038 .07124 m
.31646 .07124 L
.31646 .09499 L
.3038 .09499 L
F
.26582 .07124 m
.27848 .07124 L
.27848 .09499 L
.26582 .09499 L
F
.29114 .07124 m
.3038 .07124 L
.3038 .09499 L
.29114 .09499 L
F
.27848 .07124 m
.29114 .07124 L
.29114 .09499 L
.27848 .09499 L
F
.10127 .07124 m
.11392 .07124 L
.11392 .09499 L
.10127 .09499 L
F
.05063 .07124 m
.06329 .07124 L
.06329 .09499 L
.05063 .09499 L
F
.08861 .07124 m
.10127 .07124 L
.10127 .09499 L
.08861 .09499 L
F
.07595 .07124 m
.08861 .07124 L
.08861 .09499 L
.07595 .09499 L
F
.93671 .33245 m
.94937 .33245 L
.94937 .3562 L
.93671 .3562 L
F
.94937 .33245 m
.96203 .33245 L
.96203 .3562 L
.94937 .3562 L
F
.96203 .33245 m
.97468 .33245 L
.97468 .3562 L
.96203 .3562 L
F
.49367 .33245 m
.50633 .33245 L
.50633 .3562 L
.49367 .3562 L
F
.50633 .33245 m
.51899 .33245 L
.51899 .3562 L
.50633 .3562 L
F
.01266 .3562 m
.02532 .3562 L
.02532 .37994 L
.01266 .37994 L
F
.92405 .33245 m
.93671 .33245 L
.93671 .3562 L
.92405 .3562 L
F
0 .3562 m
.01266 .3562 L
.01266 .37994 L
0 .37994 L
F
.91139 .33245 m
.92405 .33245 L
.92405 .3562 L
.91139 .3562 L
F
.89873 .33245 m
.91139 .33245 L
.91139 .3562 L
.89873 .3562 L
F
.48101 .33245 m
.49367 .33245 L
.49367 .3562 L
.48101 .3562 L
F
.4557 .33245 m
.46835 .33245 L
.46835 .3562 L
.4557 .3562 L
F
.43038 .33245 m
.44304 .33245 L
.44304 .3562 L
.43038 .3562 L
F
.64557 .33245 m
.65823 .33245 L
.65823 .3562 L
.64557 .3562 L
F
.46835 .33245 m
.48101 .33245 L
.48101 .3562 L
.46835 .3562 L
F
.51899 .33245 m
.53165 .33245 L
.53165 .3562 L
.51899 .3562 L
F
.53165 .33245 m
.5443 .33245 L
.5443 .3562 L
.53165 .3562 L
F
.5443 .33245 m
.55696 .33245 L
.55696 .3562 L
.5443 .3562 L
F
.41772 .33245 m
.43038 .33245 L
.43038 .3562 L
.41772 .3562 L
F
.44304 .33245 m
.4557 .33245 L
.4557 .3562 L
.44304 .3562 L
F
.93671 .04749 m
.94937 .04749 L
.94937 .07124 L
.93671 .07124 L
F
.94937 .04749 m
.96203 .04749 L
.96203 .07124 L
.94937 .07124 L
F
.96203 .04749 m
.97468 .04749 L
.97468 .07124 L
.96203 .07124 L
F
.83544 .33245 m
.8481 .33245 L
.8481 .3562 L
.83544 .3562 L
F
.8481 .33245 m
.86076 .33245 L
.86076 .3562 L
.8481 .3562 L
F
.01266 .07124 m
.02532 .07124 L
.02532 .09499 L
.01266 .09499 L
F
.92405 .04749 m
.93671 .04749 L
.93671 .07124 L
.92405 .07124 L
F
0 .07124 m
.01266 .07124 L
.01266 .09499 L
0 .09499 L
F
.91139 .04749 m
.92405 .04749 L
.92405 .07124 L
.91139 .07124 L
F
.89873 .04749 m
.91139 .04749 L
.91139 .07124 L
.89873 .07124 L
F
.81013 .33245 m
.82278 .33245 L
.82278 .3562 L
.81013 .3562 L
F
.98734 .33245 m
1 .33245 L
1 .3562 L
.98734 .3562 L
F
.82278 .33245 m
.83544 .33245 L
.83544 .3562 L
.82278 .3562 L
F
.02532 .3562 m
.03797 .3562 L
.03797 .37994 L
.02532 .37994 L
F
.97468 .33245 m
.98734 .33245 L
.98734 .3562 L
.97468 .3562 L
F
.86076 .33245 m
.87342 .33245 L
.87342 .3562 L
.86076 .3562 L
F
.88608 .33245 m
.89873 .33245 L
.89873 .3562 L
.88608 .3562 L
F
.79114 .33245 m
.79747 .33245 L
.79747 .34432 L
.79114 .34432 L
F
.87342 .33245 m
.88608 .33245 L
.88608 .3562 L
.87342 .3562 L
F
.08861 .09499 m
.10127 .09499 L
.10127 .11873 L
.08861 .11873 L
F
.07595 .09499 m
.08861 .09499 L
.08861 .11873 L
.07595 .11873 L
F
.91139 .07124 m
.92405 .07124 L
.92405 .09499 L
.91139 .09499 L
F
.10127 .09499 m
.11392 .09499 L
.11392 .11873 L
.10127 .11873 L
F
.12658 .09499 m
.13924 .09499 L
.13924 .11873 L
.12658 .11873 L
F
.86076 .07124 m
.87342 .07124 L
.87342 .09499 L
.86076 .09499 L
F
.87342 .07124 m
.88608 .07124 L
.88608 .09499 L
.87342 .09499 L
F
.88608 .07124 m
.89873 .07124 L
.89873 .09499 L
.88608 .09499 L
F
.89873 .07124 m
.91139 .07124 L
.91139 .09499 L
.89873 .09499 L
F
.92405 .07124 m
.93671 .07124 L
.93671 .09499 L
.92405 .09499 L
F
.05063 .09499 m
.06329 .09499 L
.06329 .11873 L
.05063 .11873 L
F
.56962 .04749 m
.58228 .04749 L
.58228 .07124 L
.56962 .07124 L
F
.03797 .09499 m
.05063 .09499 L
.05063 .11873 L
.03797 .11873 L
F
.60759 .04749 m
.62025 .04749 L
.62025 .07124 L
.60759 .07124 L
F
.55696 .04749 m
.56962 .04749 L
.56962 .07124 L
.55696 .07124 L
F
.02532 .09499 m
.03797 .09499 L
.03797 .11873 L
.02532 .11873 L
F
0 .09499 m
.01266 .09499 L
.01266 .11873 L
0 .11873 L
F
.11392 .09499 m
.12658 .09499 L
.12658 .11873 L
.11392 .11873 L
F
.06329 .09499 m
.07595 .09499 L
.07595 .11873 L
.06329 .11873 L
F
.01266 .09499 m
.02532 .09499 L
.02532 .11873 L
.01266 .11873 L
F
.37975 .09499 m
.39241 .09499 L
.39241 .11873 L
.37975 .11873 L
F
.39241 .09499 m
.40506 .09499 L
.40506 .11873 L
.39241 .11873 L
F
.40506 .09499 m
.41772 .09499 L
.41772 .11873 L
.40506 .11873 L
F
.27848 .09499 m
.29114 .09499 L
.29114 .11873 L
.27848 .11873 L
F
.3038 .09499 m
.31646 .09499 L
.31646 .11873 L
.3038 .11873 L
F
.17722 .09499 m
.18987 .09499 L
.18987 .11873 L
.17722 .11873 L
F
.18987 .09499 m
.20253 .09499 L
.20253 .11873 L
.18987 .11873 L
F
.20253 .09499 m
.21519 .09499 L
.21519 .11873 L
.20253 .11873 L
F
.35443 .09499 m
.36709 .09499 L
.36709 .11873 L
.35443 .11873 L
F
.36709 .09499 m
.37975 .09499 L
.37975 .11873 L
.36709 .11873 L
F
.93671 .07124 m
.94937 .07124 L
.94937 .09499 L
.93671 .09499 L
F
.98734 .07124 m
1 .07124 L
1 .09499 L
.98734 .09499 L
F
.94937 .07124 m
.96203 .07124 L
.96203 .09499 L
.94937 .09499 L
F
.97468 .07124 m
.98734 .07124 L
.98734 .09499 L
.97468 .09499 L
F
.96203 .07124 m
.97468 .07124 L
.97468 .09499 L
.96203 .09499 L
F
.34177 .09499 m
.35443 .09499 L
.35443 .11873 L
.34177 .11873 L
F
.29114 .09499 m
.3038 .09499 L
.3038 .11873 L
.29114 .11873 L
F
.32911 .09499 m
.34177 .09499 L
.34177 .11873 L
.32911 .11873 L
F
.31646 .09499 m
.32911 .09499 L
.32911 .11873 L
.31646 .11873 L
F
.20253 .04749 m
.21519 .04749 L
.21519 .07124 L
.20253 .07124 L
F
.21519 .04749 m
.22785 .04749 L
.22785 .07124 L
.21519 .07124 L
F
.22785 .04749 m
.24051 .04749 L
.24051 .07124 L
.22785 .07124 L
F
.24051 .04749 m
.25316 .04749 L
.25316 .07124 L
.24051 .07124 L
F
.26582 .04749 m
.27848 .04749 L
.27848 .07124 L
.26582 .07124 L
F
.27848 .04749 m
.29114 .04749 L
.29114 .07124 L
.27848 .07124 L
F
.32911 .04749 m
.34177 .04749 L
.34177 .07124 L
.32911 .07124 L
F
.29114 .04749 m
.3038 .04749 L
.3038 .07124 L
.29114 .07124 L
F
.31646 .04749 m
.32911 .04749 L
.32911 .07124 L
.31646 .07124 L
F
.3038 .04749 m
.31646 .04749 L
.31646 .07124 L
.3038 .07124 L
F
.4557 .04749 m
.46835 .04749 L
.46835 .07124 L
.4557 .07124 L
F
.36709 .04749 m
.37975 .04749 L
.37975 .07124 L
.36709 .07124 L
F
.44304 .04749 m
.4557 .04749 L
.4557 .07124 L
.44304 .07124 L
F
.35443 .04749 m
.36709 .04749 L
.36709 .07124 L
.35443 .07124 L
F
.34177 .04749 m
.35443 .04749 L
.35443 .07124 L
.34177 .07124 L
F
.43038 .04749 m
.44304 .04749 L
.44304 .07124 L
.43038 .07124 L
F
.25316 .04749 m
.26582 .04749 L
.26582 .07124 L
.25316 .07124 L
F
.46835 .04749 m
.48101 .04749 L
.48101 .07124 L
.46835 .07124 L
F
.41772 .04749 m
.43038 .04749 L
.43038 .07124 L
.41772 .07124 L
F
.51899 .04749 m
.53165 .04749 L
.53165 .07124 L
.51899 .07124 L
F
.53165 .04749 m
.5443 .04749 L
.5443 .07124 L
.53165 .07124 L
F
.5443 .04749 m
.55696 .04749 L
.55696 .07124 L
.5443 .07124 L
F
.6962 .04749 m
.70886 .04749 L
.70886 .07124 L
.6962 .07124 L
F
.70886 .04749 m
.72152 .04749 L
.72152 .07124 L
.70886 .07124 L
F
.59494 .04749 m
.60759 .04749 L
.60759 .07124 L
.59494 .07124 L
F
.50633 .04749 m
.51899 .04749 L
.51899 .07124 L
.50633 .07124 L
F
.58228 .04749 m
.59494 .04749 L
.59494 .07124 L
.58228 .07124 L
F
.49367 .04749 m
.50633 .04749 L
.50633 .07124 L
.49367 .07124 L
F
.48101 .04749 m
.49367 .04749 L
.49367 .07124 L
.48101 .07124 L
F
.63291 .04749 m
.64557 .04749 L
.64557 .07124 L
.63291 .07124 L
F
.68354 .04749 m
.6962 .04749 L
.6962 .07124 L
.68354 .07124 L
F
.62025 .04749 m
.63291 .04749 L
.63291 .07124 L
.62025 .07124 L
F
.67089 .04749 m
.68354 .04749 L
.68354 .07124 L
.67089 .07124 L
F
.65823 .04749 m
.67089 .04749 L
.67089 .07124 L
.65823 .07124 L
F
.72152 .04749 m
.73418 .04749 L
.73418 .07124 L
.72152 .07124 L
F
.74684 .04749 m
.75949 .04749 L
.75949 .07124 L
.74684 .07124 L
F
.64557 .04749 m
.65823 .04749 L
.65823 .07124 L
.64557 .07124 L
F
.73418 .04749 m
.74684 .04749 L
.74684 .07124 L
.73418 .07124 L
F
.74684 .47493 m
.75949 .47493 L
.75949 .49867 L
.74684 .49867 L
F
.65823 .47493 m
.67089 .47493 L
.67089 .49867 L
.65823 .49867 L
F
.73418 .47493 m
.74684 .47493 L
.74684 .49867 L
.73418 .49867 L
F
.64557 .47493 m
.65823 .47493 L
.65823 .49867 L
.64557 .49867 L
F
.63291 .47493 m
.64557 .47493 L
.64557 .49867 L
.63291 .49867 L
F
.25316 .47493 m
.26582 .47493 L
.26582 .49867 L
.25316 .49867 L
F
.72152 .47493 m
.73418 .47493 L
.73418 .49867 L
.72152 .49867 L
F
.55696 .47493 m
.56962 .47493 L
.56962 .49867 L
.55696 .49867 L
F
.75949 .47493 m
.77215 .47493 L
.77215 .49867 L
.75949 .49867 L
F
.70886 .47493 m
.72152 .47493 L
.72152 .49867 L
.70886 .49867 L
F
.06329 .47493 m
.07595 .47493 L
.07595 .49867 L
.06329 .49867 L
F
.03797 .47493 m
.05063 .47493 L
.05063 .49867 L
.03797 .49867 L
F
.01266 .47493 m
.02532 .47493 L
.02532 .49867 L
.01266 .49867 L
F
.96203 .45118 m
.97468 .45118 L
.97468 .47493 L
.96203 .47493 L
F
.05063 .47493 m
.06329 .47493 L
.06329 .49867 L
.05063 .49867 L
F
.67089 .47493 m
.68354 .47493 L
.68354 .49867 L
.67089 .49867 L
F
.6962 .47493 m
.70886 .47493 L
.70886 .49867 L
.6962 .49867 L
F
.02532 .47493 m
.03797 .47493 L
.03797 .49867 L
.02532 .49867 L
F
.68354 .47493 m
.6962 .47493 L
.6962 .49867 L
.68354 .49867 L
F
.53165 .45118 m
.5443 .45118 L
.5443 .47493 L
.53165 .47493 L
F
.5443 .45118 m
.55696 .45118 L
.55696 .47493 L
.5443 .47493 L
F
.55696 .45118 m
.56962 .45118 L
.56962 .47493 L
.55696 .47493 L
F
.56962 .45118 m
.58228 .45118 L
.58228 .47493 L
.56962 .47493 L
F
.59494 .45118 m
.60759 .45118 L
.60759 .47493 L
.59494 .47493 L
F
.60759 .45118 m
.62025 .45118 L
.62025 .47493 L
.60759 .47493 L
F
.65823 .45118 m
.67089 .45118 L
.67089 .47493 L
.65823 .47493 L
F
.62025 .45118 m
.63291 .45118 L
.63291 .47493 L
.62025 .47493 L
F
.64557 .45118 m
.65823 .45118 L
.65823 .47493 L
.64557 .47493 L
F
.63291 .45118 m
.64557 .45118 L
.64557 .47493 L
.63291 .47493 L
F
.60759 .47493 m
.62025 .47493 L
.62025 .49867 L
.60759 .49867 L
F
.24051 .47493 m
.25316 .47493 L
.25316 .49867 L
.24051 .49867 L
F
.59494 .47493 m
.60759 .47493 L
.60759 .49867 L
.59494 .49867 L
F
.22785 .47493 m
.24051 .47493 L
.24051 .49867 L
.22785 .49867 L
F
.21519 .47493 m
.22785 .47493 L
.22785 .49867 L
.21519 .49867 L
F
.58228 .47493 m
.59494 .47493 L
.59494 .49867 L
.58228 .49867 L
F
.58228 .45118 m
.59494 .45118 L
.59494 .47493 L
.58228 .47493 L
F
.62025 .47493 m
.63291 .47493 L
.63291 .49867 L
.62025 .49867 L
F
.56962 .47493 m
.58228 .47493 L
.58228 .49867 L
.56962 .49867 L
F
.21519 .42744 m
.22785 .42744 L
.22785 .45118 L
.21519 .45118 L
F
.55696 .42744 m
.56962 .42744 L
.56962 .45118 L
.55696 .45118 L
F
.20253 .42744 m
.21519 .42744 L
.21519 .45118 L
.20253 .45118 L
F
.59494 .42744 m
.60759 .42744 L
.60759 .45118 L
.59494 .45118 L
F
.5443 .42744 m
.55696 .42744 L
.55696 .45118 L
.5443 .45118 L
F
.27848 .42744 m
.29114 .42744 L
.29114 .45118 L
.27848 .45118 L
F
.18987 .42744 m
.20253 .42744 L
.20253 .45118 L
.18987 .45118 L
F
.26582 .42744 m
.27848 .42744 L
.27848 .45118 L
.26582 .45118 L
F
.22785 .42744 m
.24051 .42744 L
.24051 .45118 L
.22785 .45118 L
F
.17722 .42744 m
.18987 .42744 L
.18987 .45118 L
.17722 .45118 L
F
.53165 .42744 m
.5443 .42744 L
.5443 .45118 L
.53165 .45118 L
F
.50633 .42744 m
.51899 .42744 L
.51899 .45118 L
.50633 .45118 L
F
.31646 .42744 m
.32911 .42744 L
.32911 .45118 L
.31646 .45118 L
F
.98734 .40369 m
1 .40369 L
1 .42744 L
.98734 .42744 L
F
.51899 .42744 m
.53165 .42744 L
.53165 .45118 L
.51899 .45118 L
F
.58228 .42744 m
.59494 .42744 L
.59494 .45118 L
.58228 .45118 L
F
.56962 .42744 m
.58228 .42744 L
.58228 .45118 L
.56962 .45118 L
F
.3038 .42744 m
.31646 .42744 L
.31646 .45118 L
.3038 .45118 L
F
.49367 .42744 m
.50633 .42744 L
.50633 .45118 L
.49367 .45118 L
F
.1519 .47493 m
.16456 .47493 L
.16456 .49867 L
.1519 .49867 L
F
.20253 .47493 m
.21519 .47493 L
.21519 .49867 L
.20253 .49867 L
F
.16456 .47493 m
.17722 .47493 L
.17722 .49867 L
.16456 .49867 L
F
.18987 .47493 m
.20253 .47493 L
.20253 .49867 L
.18987 .49867 L
F
.17722 .47493 m
.18987 .47493 L
.18987 .49867 L
.17722 .49867 L
F
.94937 .45118 m
.96203 .45118 L
.96203 .47493 L
.94937 .47493 L
F
0 .47493 m
.01266 .47493 L
.01266 .49867 L
0 .49867 L
F
.93671 .45118 m
.94937 .45118 L
.94937 .47493 L
.93671 .47493 L
F
.98734 .45118 m
1 .45118 L
1 .47493 L
.98734 .47493 L
F
.97468 .45118 m
.98734 .45118 L
.98734 .47493 L
.97468 .47493 L
F
.12658 .47493 m
.13924 .47493 L
.13924 .49867 L
.12658 .49867 L
F
.25316 .42744 m
.26582 .42744 L
.26582 .45118 L
.25316 .45118 L
F
.11392 .47493 m
.12658 .47493 L
.12658 .49867 L
.11392 .49867 L
F
.29114 .42744 m
.3038 .42744 L
.3038 .45118 L
.29114 .45118 L
F
.24051 .42744 m
.25316 .42744 L
.25316 .45118 L
.24051 .45118 L
F
.07595 .47493 m
.08861 .47493 L
.08861 .49867 L
.07595 .49867 L
F
.10127 .47493 m
.11392 .47493 L
.11392 .49867 L
.10127 .49867 L
F
.13924 .47493 m
.1519 .47493 L
.1519 .49867 L
.13924 .49867 L
F
.08861 .47493 m
.10127 .47493 L
.10127 .49867 L
.08861 .49867 L
F
.60759 .68865 m
.62025 .68865 L
.62025 .71239 L
.60759 .71239 L
F
.08861 .68865 m
.10127 .68865 L
.10127 .71239 L
.08861 .71239 L
F
.06329 .68865 m
.07595 .68865 L
.07595 .71239 L
.06329 .71239 L
F
.01266 .68865 m
.02532 .68865 L
.02532 .71239 L
.01266 .71239 L
F
.59854 .68865 m
.60759 .68865 L
.60759 .71239 L
.59888 .71239 L
F
.96203 .68865 m
.97468 .68865 L
.97468 .71239 L
.96203 .71239 L
F
.93671 .68865 m
.94937 .68865 L
.94937 .71239 L
.93671 .71239 L
F
.91139 .68865 m
.92405 .68865 L
.92405 .71239 L
.91139 .71239 L
F
.07595 .68865 m
.08861 .68865 L
.08861 .71239 L
.07595 .71239 L
F
.94937 .68865 m
.96203 .68865 L
.96203 .71239 L
.94937 .71239 L
F
.70886 .68865 m
.72152 .68865 L
.72152 .71239 L
.70886 .71239 L
F
.6962 .68865 m
.70886 .68865 L
.70886 .71239 L
.6962 .71239 L
F
.03797 .68865 m
.05063 .68865 L
.05063 .71239 L
.03797 .71239 L
F
.72152 .68865 m
.73418 .68865 L
.73418 .71239 L
.72152 .71239 L
F
.74684 .68865 m
.75949 .68865 L
.75949 .71239 L
.74684 .71239 L
F
0 .68865 m
.01266 .68865 L
.01266 .71239 L
0 .71239 L
F
.98734 .6649 m
1 .6649 L
1 .68865 L
.98734 .68865 L
F
.02532 .68865 m
.03797 .68865 L
.03797 .71239 L
.02532 .71239 L
F
.05063 .68865 m
.06329 .68865 L
.06329 .71239 L
.05063 .71239 L
F
.78481 .68865 m
.79747 .68865 L
.79747 .71239 L
.78481 .71239 L
F
.75949 .68865 m
.77215 .68865 L
.77215 .71239 L
.75949 .71239 L
F
.87342 .68865 m
.88608 .68865 L
.88608 .71239 L
.87342 .71239 L
F
.82278 .68865 m
.83544 .68865 L
.83544 .71239 L
.82278 .71239 L
F
.77215 .68865 m
.78481 .68865 L
.78481 .71239 L
.77215 .71239 L
F
.8481 .68865 m
.86076 .68865 L
.86076 .71239 L
.8481 .71239 L
F
.83544 .68865 m
.8481 .68865 L
.8481 .71239 L
.83544 .71239 L
F
.62025 .6649 m
.63291 .6649 L
.63291 .68865 L
.62025 .68865 L
F
.86076 .68865 m
.87342 .68865 L
.87342 .71239 L
.86076 .71239 L
F
.88608 .68865 m
.89873 .68865 L
.89873 .71239 L
.88608 .71239 L
F
0 .71239 m
.01266 .71239 L
.01266 .73614 L
0 .73614 L
F
.01266 .71239 m
.02532 .71239 L
.02532 .73614 L
.01266 .73614 L
F
.02532 .71239 m
.03797 .71239 L
.03797 .73614 L
.02532 .73614 L
F
.89873 .68865 m
.91139 .68865 L
.91139 .71239 L
.89873 .71239 L
F
.92405 .68865 m
.93671 .68865 L
.93671 .71239 L
.92405 .71239 L
F
.81013 .68865 m
.82278 .68865 L
.82278 .71239 L
.81013 .71239 L
F
.79747 .68865 m
.81013 .68865 L
.81013 .71239 L
.79747 .71239 L
F
.97468 .68865 m
.98734 .68865 L
.98734 .71239 L
.97468 .71239 L
F
.98734 .68865 m
1 .68865 L
1 .71239 L
.98734 .71239 L
F
.86076 .45118 m
.87342 .45118 L
.87342 .47493 L
.86076 .47493 L
F
.83544 .45118 m
.8481 .45118 L
.8481 .47493 L
.83544 .47493 L
F
.81013 .45118 m
.82278 .45118 L
.82278 .47493 L
.81013 .47493 L
F
.25316 .45118 m
.26582 .45118 L
.26582 .47493 L
.25316 .47493 L
F
.8481 .45118 m
.86076 .45118 L
.86076 .47493 L
.8481 .47493 L
F
.89873 .45118 m
.91139 .45118 L
.91139 .47493 L
.89873 .47493 L
F
.91139 .45118 m
.92405 .45118 L
.92405 .47493 L
.91139 .47493 L
F
.92405 .45118 m
.93671 .45118 L
.93671 .47493 L
.92405 .47493 L
F
.79747 .45118 m
.81013 .45118 L
.81013 .47493 L
.79747 .47493 L
F
.82278 .45118 m
.83544 .45118 L
.83544 .47493 L
.82278 .47493 L
F
.17722 .45118 m
.18987 .45118 L
.18987 .47493 L
.17722 .47493 L
F
.22785 .45118 m
.24051 .45118 L
.24051 .47493 L
.22785 .47493 L
F
.16456 .45118 m
.17722 .45118 L
.17722 .47493 L
.16456 .47493 L
F
.21519 .45118 m
.22785 .45118 L
.22785 .47493 L
.21519 .47493 L
F
.20253 .45118 m
.21519 .45118 L
.21519 .47493 L
.20253 .47493 L
F
.26582 .45118 m
.27848 .45118 L
.27848 .47493 L
.26582 .47493 L
F
.24051 .45118 m
.25316 .45118 L
.25316 .47493 L
.24051 .47493 L
F
.18987 .45118 m
.20253 .45118 L
.20253 .47493 L
.18987 .47493 L
F
.27848 .45118 m
.29114 .45118 L
.29114 .47493 L
.27848 .47493 L
F
.67089 .68865 m
.68354 .68865 L
.68354 .71239 L
.67089 .71239 L
F
.74684 .45118 m
.75949 .45118 L
.75949 .47493 L
.74684 .47493 L
F
.65823 .68865 m
.67089 .68865 L
.67089 .71239 L
.65823 .71239 L
F
.78481 .45118 m
.79747 .45118 L
.79747 .47493 L
.78481 .47493 L
F
.73418 .45118 m
.74684 .45118 L
.74684 .47493 L
.73418 .47493 L
F
.64557 .68865 m
.65823 .68865 L
.65823 .71239 L
.64557 .71239 L
F
.62025 .68865 m
.63291 .68865 L
.63291 .71239 L
.62025 .71239 L
F
.73418 .68865 m
.74684 .68865 L
.74684 .71239 L
.73418 .71239 L
F
.68354 .68865 m
.6962 .68865 L
.6962 .71239 L
.68354 .71239 L
F
.63291 .68865 m
.64557 .68865 L
.64557 .71239 L
.63291 .71239 L
F
.6962 .45118 m
.70886 .45118 L
.70886 .47493 L
.6962 .47493 L
F
.70886 .45118 m
.72152 .45118 L
.72152 .47493 L
.70886 .47493 L
F
.72152 .45118 m
.73418 .45118 L
.73418 .47493 L
.72152 .47493 L
F
.87342 .45118 m
.88608 .45118 L
.88608 .47493 L
.87342 .47493 L
F
.88608 .45118 m
.89873 .45118 L
.89873 .47493 L
.88608 .47493 L
F
.77215 .45118 m
.78481 .45118 L
.78481 .47493 L
.77215 .47493 L
F
.75949 .45118 m
.77215 .45118 L
.77215 .47493 L
.75949 .47493 L
F
.67089 .45118 m
.68354 .45118 L
.68354 .47493 L
.67089 .47493 L
F
.68354 .45118 m
.6962 .45118 L
.6962 .47493 L
.68354 .47493 L
F
.70886 .52242 m
.72152 .52242 L
.72152 .54617 L
.70886 .54617 L
F
.68354 .52242 m
.6962 .52242 L
.6962 .54617 L
.68354 .54617 L
F
.79747 .52242 m
.81013 .52242 L
.81013 .54617 L
.79747 .54617 L
F
.74684 .52242 m
.75949 .52242 L
.75949 .54617 L
.74684 .54617 L
F
.6962 .52242 m
.70886 .52242 L
.70886 .54617 L
.6962 .54617 L
F
.77215 .52242 m
.78481 .52242 L
.78481 .54617 L
.77215 .54617 L
F
.75949 .52242 m
.77215 .52242 L
.77215 .54617 L
.75949 .54617 L
F
.59494 .52242 m
.60759 .52242 L
.60759 .54617 L
.59494 .54617 L
F
.78481 .52242 m
.79747 .52242 L
.79747 .54617 L
.78481 .54617 L
F
.81013 .52242 m
.82278 .52242 L
.82278 .54617 L
.81013 .54617 L
F
.8481 .54617 m
.86076 .54617 L
.86076 .56991 L
.8481 .56991 L
F
.86076 .54617 m
.87342 .54617 L
.87342 .56991 L
.86076 .56991 L
F
.87342 .54617 m
.88608 .54617 L
.88608 .56991 L
.87342 .56991 L
F
.74684 .54617 m
.75949 .54617 L
.75949 .56991 L
.74684 .56991 L
F
.77215 .54617 m
.78481 .54617 L
.78481 .56991 L
.77215 .56991 L
F
.73418 .52242 m
.74684 .52242 L
.74684 .54617 L
.73418 .54617 L
F
.72152 .52242 m
.73418 .52242 L
.73418 .54617 L
.72152 .54617 L
F
.82278 .54617 m
.83544 .54617 L
.83544 .56991 L
.82278 .56991 L
F
.83544 .54617 m
.8481 .54617 L
.8481 .56991 L
.83544 .56991 L
F
.96203 .52242 m
.97468 .52242 L
.97468 .54617 L
.96203 .54617 L
F
.01266 .54617 m
.02532 .54617 L
.02532 .56991 L
.01266 .56991 L
F
.94937 .52242 m
.96203 .52242 L
.96203 .54617 L
.94937 .54617 L
F
0 .54617 m
.01266 .54617 L
.01266 .56991 L
0 .56991 L
F
.98734 .52242 m
1 .52242 L
1 .54617 L
.98734 .54617 L
F
.07595 .54617 m
.08861 .54617 L
.08861 .56991 L
.07595 .56991 L
F
.05063 .54617 m
.06329 .54617 L
.06329 .56991 L
.05063 .56991 L
F
.02532 .54617 m
.03797 .54617 L
.03797 .56991 L
.02532 .56991 L
F
.97468 .52242 m
.98734 .52242 L
.98734 .54617 L
.97468 .54617 L
F
.06329 .54617 m
.07595 .54617 L
.07595 .56991 L
.06329 .56991 L
F
.20253 .52242 m
.21519 .52242 L
.21519 .54617 L
.20253 .54617 L
F
.21519 .52242 m
.22785 .52242 L
.22785 .54617 L
.21519 .54617 L
F
.65823 .52242 m
.67089 .52242 L
.67089 .54617 L
.65823 .54617 L
F
.58228 .52242 m
.59494 .52242 L
.59494 .54617 L
.58228 .54617 L
F
.60759 .52242 m
.62025 .52242 L
.62025 .54617 L
.60759 .54617 L
F
.62025 .52242 m
.63291 .52242 L
.63291 .54617 L
.62025 .54617 L
F
.63291 .52242 m
.64557 .52242 L
.64557 .54617 L
.63291 .54617 L
F
.64557 .52242 m
.65823 .52242 L
.65823 .54617 L
.64557 .54617 L
F
.67089 .52242 m
.68354 .52242 L
.68354 .54617 L
.67089 .54617 L
F
.6962 .54617 m
.70886 .54617 L
.70886 .56991 L
.6962 .56991 L
F
.68354 .54617 m
.6962 .54617 L
.6962 .56991 L
.68354 .56991 L
F
.13924 .54617 m
.1519 .54617 L
.1519 .56991 L
.13924 .56991 L
F
.70886 .54617 m
.72152 .54617 L
.72152 .56991 L
.70886 .56991 L
F
.73418 .54617 m
.74684 .54617 L
.74684 .56991 L
.73418 .56991 L
F
.08861 .54617 m
.10127 .54617 L
.10127 .56991 L
.08861 .56991 L
F
.10127 .54617 m
.11392 .54617 L
.11392 .56991 L
.10127 .56991 L
F
.11392 .54617 m
.12658 .54617 L
.12658 .56991 L
.11392 .56991 L
F
.12658 .54617 m
.13924 .54617 L
.13924 .56991 L
.12658 .56991 L
F
.1519 .54617 m
.16456 .54617 L
.16456 .56991 L
.1519 .56991 L
F
.64557 .54617 m
.65823 .54617 L
.65823 .56991 L
.64557 .56991 L
F
.65823 .54617 m
.67089 .54617 L
.67089 .56991 L
.65823 .56991 L
F
.67089 .54617 m
.68354 .54617 L
.68354 .56991 L
.67089 .56991 L
F
.12658 .49867 m
.13924 .49867 L
.13924 .52242 L
.12658 .52242 L
F
.13924 .49867 m
.1519 .49867 L
.1519 .52242 L
.13924 .52242 L
F
.63291 .54617 m
.64557 .54617 L
.64557 .56991 L
.63291 .56991 L
F
.72152 .54617 m
.73418 .54617 L
.73418 .56991 L
.72152 .56991 L
F
.62025 .54617 m
.63291 .54617 L
.63291 .56991 L
.62025 .56991 L
F
.60759 .54617 m
.62025 .54617 L
.62025 .56991 L
.60759 .56991 L
F
.01266 .56991 m
.02532 .56991 L
.02532 .59366 L
.01266 .59366 L
F
.98734 .54617 m
1 .54617 L
1 .56991 L
.98734 .56991 L
F
.96203 .54617 m
.97468 .54617 L
.97468 .56991 L
.96203 .56991 L
F
.91139 .54617 m
.92405 .54617 L
.92405 .56991 L
.91139 .56991 L
F
0 .56991 m
.01266 .56991 L
.01266 .59366 L
0 .59366 L
F
.81013 .54617 m
.82278 .54617 L
.82278 .56991 L
.81013 .56991 L
F
.78481 .54617 m
.79747 .54617 L
.79747 .56991 L
.78481 .56991 L
F
.75949 .54617 m
.77215 .54617 L
.77215 .56991 L
.75949 .56991 L
F
.97468 .54617 m
.98734 .54617 L
.98734 .56991 L
.97468 .56991 L
F
.79747 .54617 m
.81013 .54617 L
.81013 .56991 L
.79747 .56991 L
F
.17722 .54617 m
.18987 .54617 L
.18987 .56991 L
.17722 .56991 L
F
.16456 .54617 m
.17722 .54617 L
.17722 .56991 L
.16456 .56991 L
F
.93671 .54617 m
.94937 .54617 L
.94937 .56991 L
.93671 .56991 L
F
.18987 .54617 m
.20253 .54617 L
.20253 .56991 L
.18987 .56991 L
F
.59494 .54617 m
.60759 .54617 L
.60759 .56991 L
.59494 .56991 L
F
.89873 .54617 m
.91139 .54617 L
.91139 .56991 L
.89873 .56991 L
F
.88608 .54617 m
.89873 .54617 L
.89873 .56991 L
.88608 .56991 L
F
.92405 .54617 m
.93671 .54617 L
.93671 .56991 L
.92405 .56991 L
F
.94937 .54617 m
.96203 .54617 L
.96203 .56991 L
.94937 .56991 L
F
0 .45118 m
.01266 .45118 L
.01266 .47493 L
0 .47493 L
F
.91139 .42744 m
.92405 .42744 L
.92405 .45118 L
.91139 .45118 L
F
.98734 .42744 m
1 .42744 L
1 .45118 L
.98734 .45118 L
F
.89873 .42744 m
.91139 .42744 L
.91139 .45118 L
.89873 .45118 L
F
.88608 .42744 m
.89873 .42744 L
.89873 .45118 L
.88608 .45118 L
F
.08861 .42744 m
.10127 .42744 L
.10127 .45118 L
.08861 .45118 L
F
.97468 .42744 m
.98734 .42744 L
.98734 .45118 L
.97468 .45118 L
F
.07595 .42744 m
.08861 .42744 L
.08861 .45118 L
.07595 .45118 L
F
.01266 .45118 m
.02532 .45118 L
.02532 .47493 L
.01266 .47493 L
F
.96203 .42744 m
.97468 .42744 L
.97468 .45118 L
.96203 .45118 L
F
.1519 .45118 m
.16456 .45118 L
.16456 .47493 L
.1519 .47493 L
F
.12658 .45118 m
.13924 .45118 L
.13924 .47493 L
.12658 .47493 L
F
.10127 .45118 m
.11392 .45118 L
.11392 .47493 L
.10127 .47493 L
F
.05063 .45118 m
.06329 .45118 L
.06329 .47493 L
.05063 .47493 L
F
.13924 .45118 m
.1519 .45118 L
.1519 .47493 L
.13924 .47493 L
F
.92405 .42744 m
.93671 .42744 L
.93671 .45118 L
.92405 .45118 L
F
.94937 .42744 m
.96203 .42744 L
.96203 .45118 L
.94937 .45118 L
F
.11392 .45118 m
.12658 .45118 L
.12658 .47493 L
.11392 .47493 L
F
.93671 .42744 m
.94937 .42744 L
.94937 .45118 L
.93671 .45118 L
F
.91139 .40369 m
.92405 .40369 L
.92405 .42744 L
.91139 .42744 L
F
.96203 .40369 m
.97468 .40369 L
.97468 .42744 L
.96203 .42744 L
F
.89873 .40369 m
.91139 .40369 L
.91139 .42744 L
.89873 .42744 L
F
.94937 .40369 m
.96203 .40369 L
.96203 .42744 L
.94937 .42744 L
F
.93671 .40369 m
.94937 .40369 L
.94937 .42744 L
.93671 .42744 L
F
.02532 .42744 m
.03797 .42744 L
.03797 .45118 L
.02532 .45118 L
F
0 .42744 m
.01266 .42744 L
.01266 .45118 L
0 .45118 L
F
.97468 .40369 m
.98734 .40369 L
.98734 .42744 L
.97468 .42744 L
F
.92405 .40369 m
.93671 .40369 L
.93671 .42744 L
.92405 .42744 L
F
.01266 .42744 m
.02532 .42744 L
.02532 .45118 L
.01266 .45118 L
F
.06329 .42744 m
.07595 .42744 L
.07595 .45118 L
.06329 .45118 L
F
.03797 .42744 m
.05063 .42744 L
.05063 .45118 L
.03797 .45118 L
F
.1519 .42744 m
.16456 .42744 L
.16456 .45118 L
.1519 .45118 L
F
.10127 .42744 m
.11392 .42744 L
.11392 .45118 L
.10127 .45118 L
F
.05063 .42744 m
.06329 .42744 L
.06329 .45118 L
.05063 .45118 L
F
.11392 .42744 m
.12658 .42744 L
.12658 .45118 L
.11392 .45118 L
F
.12658 .42744 m
.13924 .42744 L
.13924 .45118 L
.12658 .45118 L
F
.13924 .42744 m
.1519 .42744 L
.1519 .45118 L
.13924 .45118 L
F
.16456 .42744 m
.17722 .42744 L
.17722 .45118 L
.16456 .45118 L
F
.78481 .42744 m
.79747 .42744 L
.79747 .45118 L
.78481 .45118 L
F
.79747 .42744 m
.81013 .42744 L
.81013 .45118 L
.79747 .45118 L
F
.81013 .42744 m
.82278 .42744 L
.82278 .45118 L
.81013 .45118 L
F
.88608 .52242 m
.89873 .52242 L
.89873 .54617 L
.88608 .54617 L
F
.89873 .52242 m
.91139 .52242 L
.91139 .54617 L
.89873 .54617 L
F
.86076 .42744 m
.87342 .42744 L
.87342 .45118 L
.86076 .45118 L
F
.77215 .42744 m
.78481 .42744 L
.78481 .45118 L
.77215 .45118 L
F
.8481 .42744 m
.86076 .42744 L
.86076 .45118 L
.8481 .45118 L
F
.75949 .42744 m
.77215 .42744 L
.77215 .45118 L
.75949 .45118 L
F
.74684 .42744 m
.75949 .42744 L
.75949 .45118 L
.74684 .45118 L
F
.87342 .52242 m
.88608 .52242 L
.88608 .54617 L
.87342 .54617 L
F
.8481 .52242 m
.86076 .52242 L
.86076 .54617 L
.8481 .54617 L
F
.82278 .52242 m
.83544 .52242 L
.83544 .54617 L
.82278 .54617 L
F
.03797 .54617 m
.05063 .54617 L
.05063 .56991 L
.03797 .56991 L
F
.86076 .52242 m
.87342 .52242 L
.87342 .54617 L
.86076 .54617 L
F
.91139 .52242 m
.92405 .52242 L
.92405 .54617 L
.91139 .54617 L
F
.93671 .52242 m
.94937 .52242 L
.94937 .54617 L
.93671 .54617 L
F
.83544 .52242 m
.8481 .52242 L
.8481 .54617 L
.83544 .54617 L
F
.92405 .52242 m
.93671 .52242 L
.93671 .54617 L
.92405 .54617 L
F
.68354 .42744 m
.6962 .42744 L
.6962 .45118 L
.68354 .45118 L
F
.73418 .42744 m
.74684 .42744 L
.74684 .45118 L
.73418 .45118 L
F
.6962 .42744 m
.70886 .42744 L
.70886 .45118 L
.6962 .45118 L
F
.72152 .42744 m
.73418 .42744 L
.73418 .45118 L
.72152 .45118 L
F
.70886 .42744 m
.72152 .42744 L
.72152 .45118 L
.70886 .45118 L
F
.03797 .45118 m
.05063 .45118 L
.05063 .47493 L
.03797 .47493 L
F
.08861 .45118 m
.10127 .45118 L
.10127 .47493 L
.08861 .47493 L
F
.02532 .45118 m
.03797 .45118 L
.03797 .47493 L
.02532 .47493 L
F
.07595 .45118 m
.08861 .45118 L
.08861 .47493 L
.07595 .47493 L
F
.06329 .45118 m
.07595 .45118 L
.07595 .47493 L
.06329 .47493 L
F
.65823 .42744 m
.67089 .42744 L
.67089 .45118 L
.65823 .45118 L
F
.83544 .42744 m
.8481 .42744 L
.8481 .45118 L
.83544 .45118 L
F
.64557 .42744 m
.65823 .42744 L
.65823 .45118 L
.64557 .45118 L
F
.87342 .42744 m
.88608 .42744 L
.88608 .45118 L
.87342 .45118 L
F
.82278 .42744 m
.83544 .42744 L
.83544 .45118 L
.82278 .45118 L
F
.60759 .42744 m
.62025 .42744 L
.62025 .45118 L
.60759 .45118 L
F
.63291 .42744 m
.64557 .42744 L
.64557 .45118 L
.63291 .45118 L
F
.67089 .42744 m
.68354 .42744 L
.68354 .45118 L
.67089 .45118 L
F
.62025 .42744 m
.63291 .42744 L
.63291 .45118 L
.62025 .45118 L
F
.8481 .56991 m
.86076 .56991 L
.86076 .59366 L
.8481 .59366 L
F
.82278 .56991 m
.83544 .56991 L
.83544 .59366 L
.82278 .59366 L
F
.93671 .56991 m
.94937 .56991 L
.94937 .59366 L
.93671 .59366 L
F
.88608 .56991 m
.89873 .56991 L
.89873 .59366 L
.88608 .59366 L
F
.83544 .56991 m
.8481 .56991 L
.8481 .59366 L
.83544 .59366 L
F
.91139 .56991 m
.92405 .56991 L
.92405 .59366 L
.91139 .59366 L
F
.89873 .56991 m
.91139 .56991 L
.91139 .59366 L
.89873 .59366 L
F
.73418 .56991 m
.74684 .56991 L
.74684 .59366 L
.73418 .59366 L
F
.92405 .56991 m
.93671 .56991 L
.93671 .59366 L
.92405 .59366 L
F
.94937 .56991 m
.96203 .56991 L
.96203 .59366 L
.94937 .59366 L
F
.12658 .56991 m
.13924 .56991 L
.13924 .59366 L
.12658 .59366 L
F
.13924 .56991 m
.1519 .56991 L
.1519 .59366 L
.13924 .59366 L
F
.1519 .56991 m
.16456 .56991 L
.16456 .59366 L
.1519 .59366 L
F
.02532 .56991 m
.03797 .56991 L
.03797 .59366 L
.02532 .59366 L
F
.05063 .56991 m
.06329 .56991 L
.06329 .59366 L
.05063 .59366 L
F
.87342 .56991 m
.88608 .56991 L
.88608 .59366 L
.87342 .59366 L
F
.86076 .56991 m
.87342 .56991 L
.87342 .59366 L
.86076 .59366 L
F
.10127 .56991 m
.11392 .56991 L
.11392 .59366 L
.10127 .59366 L
F
.11392 .56991 m
.12658 .56991 L
.12658 .59366 L
.11392 .59366 L
F
.07595 .64115 m
.08861 .64115 L
.08861 .6649 L
.07595 .6649 L
F
.59666 .64115 m
.60759 .64115 L
.60759 .6649 L
.59786 .6649 L
F
.06329 .64115 m
.07595 .64115 L
.07595 .6649 L
.06329 .6649 L
F
.11392 .64115 m
.12658 .64115 L
.12658 .6649 L
.11392 .6649 L
F
.10127 .64115 m
.11392 .64115 L
.11392 .6649 L
.10127 .6649 L
F
.65823 .64115 m
.67089 .64115 L
.67089 .6649 L
.65823 .6649 L
F
.63291 .64115 m
.64557 .64115 L
.64557 .6649 L
.63291 .6649 L
F
.60759 .64115 m
.62025 .64115 L
.62025 .6649 L
.60759 .6649 L
F
.08861 .64115 m
.10127 .64115 L
.10127 .6649 L
.08861 .6649 L
F
.64557 .64115 m
.65823 .64115 L
.65823 .6649 L
.64557 .6649 L
F
.70886 .56991 m
.72152 .56991 L
.72152 .59366 L
.70886 .59366 L
F
.6962 .56991 m
.70886 .56991 L
.70886 .59366 L
.6962 .59366 L
F
.79747 .56991 m
.81013 .56991 L
.81013 .59366 L
.79747 .59366 L
F
.72152 .56991 m
.73418 .56991 L
.73418 .59366 L
.72152 .59366 L
F
.74684 .56991 m
.75949 .56991 L
.75949 .59366 L
.74684 .59366 L
F
.75949 .56991 m
.77215 .56991 L
.77215 .59366 L
.75949 .59366 L
F
.77215 .56991 m
.78481 .56991 L
.78481 .59366 L
.77215 .59366 L
F
.78481 .56991 m
.79747 .56991 L
.79747 .59366 L
.78481 .59366 L
F
.81013 .56991 m
.82278 .56991 L
.82278 .59366 L
.81013 .59366 L
F
.70886 .59366 m
.72152 .59366 L
.72152 .61741 L
.70886 .61741 L
F
.88608 .59366 m
.89873 .59366 L
.89873 .61741 L
.88608 .61741 L
F
.6962 .59366 m
.70886 .59366 L
.70886 .61741 L
.6962 .61741 L
F
.92405 .59366 m
.93671 .59366 L
.93671 .61741 L
.92405 .61741 L
F
.87342 .59366 m
.88608 .59366 L
.88608 .61741 L
.87342 .61741 L
F
.68354 .59366 m
.6962 .59366 L
.6962 .61741 L
.68354 .61741 L
F
.65823 .59366 m
.67089 .59366 L
.67089 .61741 L
.65823 .61741 L
F
.77215 .59366 m
.78481 .59366 L
.78481 .61741 L
.77215 .61741 L
F
.72152 .59366 m
.73418 .59366 L
.73418 .61741 L
.72152 .61741 L
F
.67089 .59366 m
.68354 .59366 L
.68354 .61741 L
.67089 .61741 L
F
.86076 .59366 m
.87342 .59366 L
.87342 .61741 L
.86076 .61741 L
F
.83544 .59366 m
.8481 .59366 L
.8481 .61741 L
.83544 .61741 L
F
.81013 .59366 m
.82278 .59366 L
.82278 .61741 L
.81013 .61741 L
F
.05063 .59366 m
.06329 .59366 L
.06329 .61741 L
.05063 .61741 L
F
.8481 .59366 m
.86076 .59366 L
.86076 .61741 L
.8481 .61741 L
F
.91139 .59366 m
.92405 .59366 L
.92405 .61741 L
.91139 .61741 L
F
.89873 .59366 m
.91139 .59366 L
.91139 .61741 L
.89873 .61741 L
F
.79747 .59366 m
.81013 .59366 L
.81013 .61741 L
.79747 .61741 L
F
.82278 .59366 m
.83544 .59366 L
.83544 .61741 L
.82278 .61741 L
F
.68354 .56991 m
.6962 .56991 L
.6962 .59366 L
.68354 .59366 L
F
.65823 .56991 m
.67089 .56991 L
.67089 .59366 L
.65823 .59366 L
F
.63291 .56991 m
.64557 .56991 L
.64557 .59366 L
.63291 .59366 L
F
.58639 .56991 m
.59494 .56991 L
.59494 .59366 L
.59135 .59366 L
F
.67089 .56991 m
.68354 .56991 L
.68354 .59366 L
.67089 .59366 L
F
.08861 .56991 m
.10127 .56991 L
.10127 .59366 L
.08861 .59366 L
F
.06329 .56991 m
.07595 .56991 L
.07595 .59366 L
.06329 .59366 L
F
.03797 .56991 m
.05063 .56991 L
.05063 .59366 L
.03797 .59366 L
F
.64557 .56991 m
.65823 .56991 L
.65823 .59366 L
.64557 .59366 L
F
.07595 .56991 m
.08861 .56991 L
.08861 .59366 L
.07595 .59366 L
F
.74684 .59366 m
.75949 .59366 L
.75949 .61741 L
.74684 .61741 L
F
.73418 .59366 m
.74684 .59366 L
.74684 .61741 L
.73418 .61741 L
F
.60759 .56991 m
.62025 .56991 L
.62025 .59366 L
.60759 .59366 L
F
.75949 .59366 m
.77215 .59366 L
.77215 .61741 L
.75949 .61741 L
F
.78481 .59366 m
.79747 .59366 L
.79747 .61741 L
.78481 .61741 L
F
.17722 .56991 m
.18987 .56991 L
.18987 .59366 L
.17722 .59366 L
F
.16456 .56991 m
.17722 .56991 L
.17722 .59366 L
.16456 .59366 L
F
.59494 .56991 m
.60759 .56991 L
.60759 .59366 L
.59494 .59366 L
F
.62025 .56991 m
.63291 .56991 L
.63291 .59366 L
.62025 .59366 L
F
.05063 .61741 m
.06329 .61741 L
.06329 .64115 L
.05063 .64115 L
F
.96203 .59366 m
.97468 .59366 L
.97468 .61741 L
.96203 .61741 L
F
.03797 .61741 m
.05063 .61741 L
.05063 .64115 L
.03797 .64115 L
F
.94937 .59366 m
.96203 .59366 L
.96203 .61741 L
.94937 .61741 L
F
.93671 .59366 m
.94937 .59366 L
.94937 .61741 L
.93671 .61741 L
F
.83544 .61741 m
.8481 .61741 L
.8481 .64115 L
.83544 .64115 L
F
.02532 .61741 m
.03797 .61741 L
.03797 .64115 L
.02532 .64115 L
F
.82278 .61741 m
.83544 .61741 L
.83544 .64115 L
.82278 .64115 L
F
.06329 .61741 m
.07595 .61741 L
.07595 .64115 L
.06329 .64115 L
F
.01266 .61741 m
.02532 .61741 L
.02532 .64115 L
.01266 .64115 L
F
.62025 .61741 m
.63291 .61741 L
.63291 .64115 L
.62025 .64115 L
F
.63291 .61741 m
.64557 .61741 L
.64557 .64115 L
.63291 .64115 L
F
.64557 .61741 m
.65823 .61741 L
.65823 .64115 L
.64557 .64115 L
F
.07595 .61741 m
.08861 .61741 L
.08861 .64115 L
.07595 .64115 L
F
.10127 .61741 m
.11392 .61741 L
.11392 .64115 L
.10127 .64115 L
F
.97468 .59366 m
.98734 .59366 L
.98734 .61741 L
.97468 .61741 L
F
0 .61741 m
.01266 .61741 L
.01266 .64115 L
0 .64115 L
F
.60759 .61741 m
.62025 .61741 L
.62025 .64115 L
.60759 .64115 L
F
.98734 .59366 m
1 .59366 L
1 .61741 L
.98734 .61741 L
F
.65823 .61741 m
.67089 .61741 L
.67089 .64115 L
.65823 .64115 L
F
.70886 .61741 m
.72152 .61741 L
.72152 .64115 L
.70886 .64115 L
F
.67089 .61741 m
.68354 .61741 L
.68354 .64115 L
.67089 .64115 L
F
.6962 .61741 m
.70886 .61741 L
.70886 .64115 L
.6962 .64115 L
F
.68354 .61741 m
.6962 .61741 L
.6962 .64115 L
.68354 .64115 L
F
.72152 .61741 m
.73418 .61741 L
.73418 .64115 L
.72152 .64115 L
F
.77215 .61741 m
.78481 .61741 L
.78481 .64115 L
.77215 .64115 L
F
.73418 .61741 m
.74684 .61741 L
.74684 .64115 L
.73418 .64115 L
F
.75949 .61741 m
.77215 .61741 L
.77215 .64115 L
.75949 .64115 L
F
.74684 .61741 m
.75949 .61741 L
.75949 .64115 L
.74684 .64115 L
F
.81013 .61741 m
.82278 .61741 L
.82278 .64115 L
.81013 .64115 L
F
.78481 .61741 m
.79747 .61741 L
.79747 .64115 L
.78481 .64115 L
F
.89873 .61741 m
.91139 .61741 L
.91139 .64115 L
.89873 .64115 L
F
.8481 .61741 m
.86076 .61741 L
.86076 .64115 L
.8481 .64115 L
F
.79747 .61741 m
.81013 .61741 L
.81013 .64115 L
.79747 .64115 L
F
.86076 .61741 m
.87342 .61741 L
.87342 .64115 L
.86076 .64115 L
F
.87342 .61741 m
.88608 .61741 L
.88608 .64115 L
.87342 .64115 L
F
.88608 .61741 m
.89873 .61741 L
.89873 .64115 L
.88608 .64115 L
F
.91139 .61741 m
.92405 .61741 L
.92405 .64115 L
.91139 .64115 L
F
.86076 .64115 m
.87342 .64115 L
.87342 .6649 L
.86076 .6649 L
F
.01266 .64115 m
.02532 .64115 L
.02532 .6649 L
.01266 .6649 L
F
.8481 .64115 m
.86076 .64115 L
.86076 .6649 L
.8481 .6649 L
F
.05063 .64115 m
.06329 .64115 L
.06329 .6649 L
.05063 .6649 L
F
0 .64115 m
.01266 .64115 L
.01266 .6649 L
0 .6649 L
F
.83544 .64115 m
.8481 .64115 L
.8481 .6649 L
.83544 .6649 L
F
.81013 .64115 m
.82278 .64115 L
.82278 .6649 L
.81013 .6649 L
F
.92405 .64115 m
.93671 .64115 L
.93671 .6649 L
.92405 .6649 L
F
.87342 .64115 m
.88608 .64115 L
.88608 .6649 L
.87342 .6649 L
F
.82278 .64115 m
.83544 .64115 L
.83544 .6649 L
.82278 .6649 L
F
.98734 .61741 m
1 .61741 L
1 .64115 L
.98734 .64115 L
F
.96203 .61741 m
.97468 .61741 L
.97468 .64115 L
.96203 .64115 L
F
.93671 .61741 m
.94937 .61741 L
.94937 .64115 L
.93671 .64115 L
F
.62025 .64115 m
.63291 .64115 L
.63291 .6649 L
.62025 .6649 L
F
.97468 .61741 m
.98734 .61741 L
.98734 .64115 L
.97468 .64115 L
F
.03797 .64115 m
.05063 .64115 L
.05063 .6649 L
.03797 .6649 L
F
.02532 .64115 m
.03797 .64115 L
.03797 .6649 L
.02532 .6649 L
F
.92405 .61741 m
.93671 .61741 L
.93671 .64115 L
.92405 .64115 L
F
.94937 .61741 m
.96203 .61741 L
.96203 .64115 L
.94937 .64115 L
F
.79747 .64115 m
.81013 .64115 L
.81013 .6649 L
.79747 .6649 L
F
.77215 .64115 m
.78481 .64115 L
.78481 .6649 L
.77215 .6649 L
F
.74684 .64115 m
.75949 .64115 L
.75949 .6649 L
.74684 .6649 L
F
.6962 .64115 m
.70886 .64115 L
.70886 .6649 L
.6962 .6649 L
F
.78481 .64115 m
.79747 .64115 L
.79747 .6649 L
.78481 .6649 L
F
.13924 .61741 m
.1519 .61741 L
.1519 .64115 L
.13924 .64115 L
F
.11392 .61741 m
.12658 .61741 L
.12658 .64115 L
.11392 .64115 L
F
.08861 .61741 m
.10127 .61741 L
.10127 .64115 L
.08861 .64115 L
F
.75949 .64115 m
.77215 .64115 L
.77215 .6649 L
.75949 .6649 L
F
.12658 .61741 m
.13924 .61741 L
.13924 .64115 L
.12658 .64115 L
F
.89873 .64115 m
.91139 .64115 L
.91139 .6649 L
.89873 .6649 L
F
.88608 .64115 m
.89873 .64115 L
.89873 .6649 L
.88608 .6649 L
F
.72152 .64115 m
.73418 .64115 L
.73418 .6649 L
.72152 .6649 L
F
.91139 .64115 m
.92405 .64115 L
.92405 .6649 L
.91139 .6649 L
F
.93671 .64115 m
.94937 .64115 L
.94937 .6649 L
.93671 .6649 L
F
.68354 .64115 m
.6962 .64115 L
.6962 .6649 L
.68354 .6649 L
F
.67089 .64115 m
.68354 .64115 L
.68354 .6649 L
.67089 .6649 L
F
.70886 .64115 m
.72152 .64115 L
.72152 .6649 L
.70886 .6649 L
F
.73418 .64115 m
.74684 .64115 L
.74684 .6649 L
.73418 .6649 L
F
.8481 .73614 m
.86076 .73614 L
.86076 .75989 L
.8481 .75989 L
F
.86076 .73614 m
.87342 .73614 L
.87342 .75989 L
.86076 .75989 L
F
.87342 .73614 m
.88608 .73614 L
.88608 .75989 L
.87342 .75989 L
F
.74684 .73614 m
.75949 .73614 L
.75949 .75989 L
.74684 .75989 L
F
.77215 .73614 m
.78481 .73614 L
.78481 .75989 L
.77215 .75989 L
F
.64557 .73614 m
.65823 .73614 L
.65823 .75989 L
.64557 .75989 L
F
.65823 .73614 m
.67089 .73614 L
.67089 .75989 L
.65823 .75989 L
F
.67089 .73614 m
.68354 .73614 L
.68354 .75989 L
.67089 .75989 L
F
.82278 .73614 m
.83544 .73614 L
.83544 .75989 L
.82278 .75989 L
F
.83544 .73614 m
.8481 .73614 L
.8481 .75989 L
.83544 .75989 L
F
.78481 .6649 m
.79747 .6649 L
.79747 .68865 L
.78481 .68865 L
F
.83544 .6649 m
.8481 .6649 L
.8481 .68865 L
.83544 .68865 L
F
.79747 .6649 m
.81013 .6649 L
.81013 .68865 L
.79747 .68865 L
F
.82278 .6649 m
.83544 .6649 L
.83544 .68865 L
.82278 .68865 L
F
.81013 .6649 m
.82278 .6649 L
.82278 .68865 L
.81013 .68865 L
F
.81013 .73614 m
.82278 .73614 L
.82278 .75989 L
.81013 .75989 L
F
.75949 .73614 m
.77215 .73614 L
.77215 .75989 L
.75949 .75989 L
F
.79747 .73614 m
.81013 .73614 L
.81013 .75989 L
.79747 .75989 L
F
.78481 .73614 m
.79747 .73614 L
.79747 .75989 L
.78481 .75989 L
F
.02532 .75989 m
.03797 .75989 L
.03797 .78363 L
.02532 .78363 L
F
.03797 .75989 m
.05063 .75989 L
.05063 .78363 L
.03797 .78363 L
F
.05063 .75989 m
.06329 .75989 L
.06329 .78363 L
.05063 .78363 L
F
.06329 .75989 m
.07595 .75989 L
.07595 .78363 L
.06329 .78363 L
F
.60759 .75989 m
.62025 .75989 L
.62025 .78363 L
.60759 .78363 L
F
.62025 .75989 m
.63291 .75989 L
.63291 .78363 L
.62025 .78363 L
F
.67089 .75989 m
.68354 .75989 L
.68354 .78363 L
.67089 .78363 L
F
.63291 .75989 m
.64557 .75989 L
.64557 .78363 L
.63291 .78363 L
F
.65823 .75989 m
.67089 .75989 L
.67089 .78363 L
.65823 .78363 L
F
.64557 .75989 m
.65823 .75989 L
.65823 .78363 L
.64557 .78363 L
F
.72152 .73614 m
.73418 .73614 L
.73418 .75989 L
.72152 .75989 L
F
.63291 .73614 m
.64557 .73614 L
.64557 .75989 L
.63291 .75989 L
F
.70886 .73614 m
.72152 .73614 L
.72152 .75989 L
.70886 .75989 L
F
.62025 .73614 m
.63291 .73614 L
.63291 .75989 L
.62025 .75989 L
F
.60759 .73614 m
.62025 .73614 L
.62025 .75989 L
.60759 .75989 L
F
.6962 .73614 m
.70886 .73614 L
.70886 .75989 L
.6962 .75989 L
F
.5991 .75989 m
.60759 .75989 L
.60759 .78363 L
.59911 .78363 L
F
.73418 .73614 m
.74684 .73614 L
.74684 .75989 L
.73418 .75989 L
F
.68354 .73614 m
.6962 .73614 L
.6962 .75989 L
.68354 .75989 L
F
.01266 .6649 m
.02532 .6649 L
.02532 .68865 L
.01266 .68865 L
F
.98734 .64115 m
1 .64115 L
1 .6649 L
.98734 .6649 L
F
.96203 .64115 m
.97468 .64115 L
.97468 .6649 L
.96203 .6649 L
F
.65823 .6649 m
.67089 .6649 L
.67089 .68865 L
.65823 .68865 L
F
0 .6649 m
.01266 .6649 L
.01266 .68865 L
0 .68865 L
F
.05063 .6649 m
.06329 .6649 L
.06329 .68865 L
.05063 .68865 L
F
.06329 .6649 m
.07595 .6649 L
.07595 .68865 L
.06329 .68865 L
F
.07595 .6649 m
.08861 .6649 L
.08861 .68865 L
.07595 .68865 L
F
.94937 .64115 m
.96203 .64115 L
.96203 .6649 L
.94937 .6649 L
F
.97468 .64115 m
.98734 .64115 L
.98734 .6649 L
.97468 .6649 L
F
.08861 .6649 m
.10127 .6649 L
.10127 .68865 L
.08861 .68865 L
F
.10127 .6649 m
.11392 .6649 L
.11392 .68865 L
.10127 .68865 L
F
.59786 .6649 m
.60759 .6649 L
.60759 .68865 L
.59854 .68865 L
F
.60759 .6649 m
.62025 .6649 L
.62025 .68865 L
.60759 .68865 L
F
.63291 .6649 m
.64557 .6649 L
.64557 .68865 L
.63291 .68865 L
F
.6962 .6649 m
.70886 .6649 L
.70886 .68865 L
.6962 .68865 L
F
.64557 .6649 m
.65823 .6649 L
.65823 .68865 L
.64557 .68865 L
F
.68354 .6649 m
.6962 .6649 L
.6962 .68865 L
.68354 .68865 L
F
.67089 .6649 m
.68354 .6649 L
.68354 .68865 L
.67089 .68865 L
F
.93671 .6649 m
.94937 .6649 L
.94937 .68865 L
.93671 .68865 L
F
.92405 .6649 m
.93671 .6649 L
.93671 .68865 L
.92405 .68865 L
F
.75949 .6649 m
.77215 .6649 L
.77215 .68865 L
.75949 .68865 L
F
.94937 .6649 m
.96203 .6649 L
.96203 .68865 L
.94937 .68865 L
F
.97468 .6649 m
.98734 .6649 L
.98734 .68865 L
.97468 .68865 L
F
.70886 .6649 m
.72152 .6649 L
.72152 .68865 L
.70886 .68865 L
F
.72152 .6649 m
.73418 .6649 L
.73418 .68865 L
.72152 .68865 L
F
.73418 .6649 m
.74684 .6649 L
.74684 .68865 L
.73418 .68865 L
F
.74684 .6649 m
.75949 .6649 L
.75949 .68865 L
.74684 .68865 L
F
.77215 .6649 m
.78481 .6649 L
.78481 .68865 L
.77215 .68865 L
F
.88608 .6649 m
.89873 .6649 L
.89873 .68865 L
.88608 .68865 L
F
.89873 .6649 m
.91139 .6649 L
.91139 .68865 L
.89873 .68865 L
F
.91139 .6649 m
.92405 .6649 L
.92405 .68865 L
.91139 .68865 L
F
.02532 .6649 m
.03797 .6649 L
.03797 .68865 L
.02532 .68865 L
F
.03797 .6649 m
.05063 .6649 L
.05063 .68865 L
.03797 .68865 L
F
.87342 .6649 m
.88608 .6649 L
.88608 .68865 L
.87342 .68865 L
F
.96203 .6649 m
.97468 .6649 L
.97468 .68865 L
.96203 .68865 L
F
.86076 .6649 m
.87342 .6649 L
.87342 .68865 L
.86076 .68865 L
F
.8481 .6649 m
.86076 .6649 L
.86076 .68865 L
.8481 .68865 L
F
.92405 .71239 m
.93671 .71239 L
.93671 .73614 L
.92405 .73614 L
F
.83544 .71239 m
.8481 .71239 L
.8481 .73614 L
.83544 .73614 L
F
.91139 .71239 m
.92405 .71239 L
.92405 .73614 L
.91139 .73614 L
F
.87342 .71239 m
.88608 .71239 L
.88608 .73614 L
.87342 .73614 L
F
.82278 .71239 m
.83544 .71239 L
.83544 .73614 L
.82278 .73614 L
F
.1519 .59366 m
.16456 .59366 L
.16456 .61741 L
.1519 .61741 L
F
.89873 .71239 m
.91139 .71239 L
.91139 .73614 L
.89873 .73614 L
F
.13924 .59366 m
.1519 .59366 L
.1519 .61741 L
.13924 .61741 L
F
.93671 .71239 m
.94937 .71239 L
.94937 .73614 L
.93671 .73614 L
F
.88608 .71239 m
.89873 .71239 L
.89873 .73614 L
.88608 .73614 L
F
.05063 .73614 m
.06329 .73614 L
.06329 .75989 L
.05063 .75989 L
F
.06329 .73614 m
.07595 .73614 L
.07595 .75989 L
.06329 .75989 L
F
.59904 .73614 m
.60759 .73614 L
.60759 .75989 L
.5991 .75989 L
F
.94937 .71239 m
.96203 .71239 L
.96203 .73614 L
.94937 .73614 L
F
.97468 .71239 m
.98734 .71239 L
.98734 .73614 L
.97468 .73614 L
F
.86076 .71239 m
.87342 .71239 L
.87342 .73614 L
.86076 .73614 L
F
.8481 .71239 m
.86076 .71239 L
.86076 .73614 L
.8481 .73614 L
F
.02532 .73614 m
.03797 .73614 L
.03797 .75989 L
.02532 .75989 L
F
.03797 .73614 m
.05063 .73614 L
.05063 .75989 L
.03797 .75989 L
F
.97468 .56991 m
.98734 .56991 L
.98734 .59366 L
.97468 .59366 L
F
.02532 .59366 m
.03797 .59366 L
.03797 .61741 L
.02532 .61741 L
F
.96203 .56991 m
.97468 .56991 L
.97468 .59366 L
.96203 .59366 L
F
.01266 .59366 m
.02532 .59366 L
.02532 .61741 L
.01266 .61741 L
F
0 .59366 m
.01266 .59366 L
.01266 .61741 L
0 .61741 L
F
.08861 .59366 m
.10127 .59366 L
.10127 .61741 L
.08861 .61741 L
F
.06329 .59366 m
.07595 .59366 L
.07595 .61741 L
.06329 .61741 L
F
.03797 .59366 m
.05063 .59366 L
.05063 .61741 L
.03797 .61741 L
F
.98734 .56991 m
1 .56991 L
1 .59366 L
.98734 .59366 L
F
.07595 .59366 m
.08861 .59366 L
.08861 .61741 L
.07595 .61741 L
F
.12658 .59366 m
.13924 .59366 L
.13924 .61741 L
.12658 .61741 L
F
.10127 .59366 m
.11392 .59366 L
.11392 .61741 L
.10127 .61741 L
F
.63291 .59366 m
.64557 .59366 L
.64557 .61741 L
.63291 .61741 L
F
.59135 .59366 m
.59494 .59366 L
.59494 .61741 L
.59465 .61741 L
F
.11392 .59366 m
.12658 .59366 L
.12658 .61741 L
.11392 .61741 L
F
.59494 .59366 m
.60759 .59366 L
.60759 .61741 L
.59494 .61741 L
F
.60759 .59366 m
.62025 .59366 L
.62025 .61741 L
.60759 .61741 L
F
.62025 .59366 m
.63291 .59366 L
.63291 .61741 L
.62025 .61741 L
F
.64557 .59366 m
.65823 .59366 L
.65823 .61741 L
.64557 .61741 L
F
.72152 .71239 m
.73418 .71239 L
.73418 .73614 L
.72152 .73614 L
F
.73418 .71239 m
.74684 .71239 L
.74684 .73614 L
.73418 .73614 L
F
.74684 .71239 m
.75949 .71239 L
.75949 .73614 L
.74684 .73614 L
F
.96203 .73614 m
.97468 .73614 L
.97468 .75989 L
.96203 .75989 L
F
.97468 .73614 m
.98734 .73614 L
.98734 .75989 L
.97468 .75989 L
F
.79747 .71239 m
.81013 .71239 L
.81013 .73614 L
.79747 .73614 L
F
.70886 .71239 m
.72152 .71239 L
.72152 .73614 L
.70886 .73614 L
F
.78481 .71239 m
.79747 .71239 L
.79747 .73614 L
.78481 .73614 L
F
.6962 .71239 m
.70886 .71239 L
.70886 .73614 L
.6962 .73614 L
F
.68354 .71239 m
.6962 .71239 L
.6962 .73614 L
.68354 .73614 L
F
.89873 .73614 m
.91139 .73614 L
.91139 .75989 L
.89873 .75989 L
F
.94937 .73614 m
.96203 .73614 L
.96203 .75989 L
.94937 .75989 L
F
.88608 .73614 m
.89873 .73614 L
.89873 .75989 L
.88608 .75989 L
F
.93671 .73614 m
.94937 .73614 L
.94937 .75989 L
.93671 .75989 L
F
.92405 .73614 m
.93671 .73614 L
.93671 .75989 L
.92405 .75989 L
F
.98734 .73614 m
1 .73614 L
1 .75989 L
.98734 .75989 L
F
.01266 .75989 m
.02532 .75989 L
.02532 .78363 L
.01266 .78363 L
F
.91139 .73614 m
.92405 .73614 L
.92405 .75989 L
.91139 .75989 L
F
0 .75989 m
.01266 .75989 L
.01266 .78363 L
0 .78363 L
F
.62025 .71239 m
.63291 .71239 L
.63291 .73614 L
.62025 .73614 L
F
.67089 .71239 m
.68354 .71239 L
.68354 .73614 L
.67089 .73614 L
F
.63291 .71239 m
.64557 .71239 L
.64557 .73614 L
.63291 .73614 L
F
.65823 .71239 m
.67089 .71239 L
.67089 .73614 L
.65823 .73614 L
F
.64557 .71239 m
.65823 .71239 L
.65823 .73614 L
.64557 .73614 L
F
.01266 .73614 m
.02532 .73614 L
.02532 .75989 L
.01266 .75989 L
F
.96203 .71239 m
.97468 .71239 L
.97468 .73614 L
.96203 .73614 L
F
0 .73614 m
.01266 .73614 L
.01266 .75989 L
0 .75989 L
F
.98734 .71239 m
1 .71239 L
1 .73614 L
.98734 .73614 L
F
.59888 .71239 m
.60759 .71239 L
.60759 .73614 L
.59904 .73614 L
F
.77215 .71239 m
.78481 .71239 L
.78481 .73614 L
.77215 .73614 L
F
.07595 .71239 m
.08861 .71239 L
.08861 .73614 L
.07595 .73614 L
F
.81013 .71239 m
.82278 .71239 L
.82278 .73614 L
.81013 .73614 L
F
.75949 .71239 m
.77215 .71239 L
.77215 .73614 L
.75949 .73614 L
F
.03797 .71239 m
.05063 .71239 L
.05063 .73614 L
.03797 .73614 L
F
.06329 .71239 m
.07595 .71239 L
.07595 .73614 L
.06329 .73614 L
F
.60759 .71239 m
.62025 .71239 L
.62025 .73614 L
.60759 .73614 L
F
.05063 .71239 m
.06329 .71239 L
.06329 .73614 L
.05063 .73614 L
F
.75949 1.13983 m
.77215 1.13983 L
.77215 1.16357 L
.75949 1.16357 L
F
.81013 1.13983 m
.82278 1.13983 L
.82278 1.16357 L
.81013 1.16357 L
F
.74684 1.13983 m
.75949 1.13983 L
.75949 1.16357 L
.74684 1.16357 L
F
.79747 1.13983 m
.81013 1.13983 L
.81013 1.16357 L
.79747 1.16357 L
F
.78481 1.13983 m
.79747 1.13983 L
.79747 1.16357 L
.78481 1.16357 L
F
.87342 1.13983 m
.88608 1.13983 L
.88608 1.16357 L
.87342 1.16357 L
F
.8481 1.13983 m
.86076 1.13983 L
.86076 1.16357 L
.8481 1.16357 L
F
.82278 1.13983 m
.83544 1.13983 L
.83544 1.16357 L
.82278 1.16357 L
F
.77215 1.13983 m
.78481 1.13983 L
.78481 1.16357 L
.77215 1.16357 L
F
.86076 1.13983 m
.87342 1.13983 L
.87342 1.16357 L
.86076 1.16357 L
F
.86076 1.11608 m
.87342 1.11608 L
.87342 1.13983 L
.86076 1.13983 L
F
.83544 1.11608 m
.8481 1.11608 L
.8481 1.13983 L
.83544 1.13983 L
F
.94937 1.11608 m
.96203 1.11608 L
.96203 1.13983 L
.94937 1.13983 L
F
.89873 1.11608 m
.91139 1.11608 L
.91139 1.13983 L
.89873 1.13983 L
F
.8481 1.11608 m
.86076 1.11608 L
.86076 1.13983 L
.8481 1.13983 L
F
.91139 1.11608 m
.92405 1.11608 L
.92405 1.13983 L
.91139 1.13983 L
F
.92405 1.11608 m
.93671 1.11608 L
.93671 1.13983 L
.92405 1.13983 L
F
.93671 1.11608 m
.94937 1.11608 L
.94937 1.13983 L
.93671 1.13983 L
F
.96203 1.11608 m
.97468 1.11608 L
.97468 1.13983 L
.96203 1.13983 L
F
.96203 1.09233 m
.97468 1.09233 L
.97468 1.11608 L
.96203 1.11608 L
F
.6962 1.13983 m
.70886 1.13983 L
.70886 1.16357 L
.6962 1.16357 L
F
.94937 1.09233 m
.96203 1.09233 L
.96203 1.11608 L
.94937 1.11608 L
F
.73418 1.13983 m
.74684 1.13983 L
.74684 1.16357 L
.73418 1.16357 L
F
.68354 1.13983 m
.6962 1.13983 L
.6962 1.16357 L
.68354 1.16357 L
F
.93671 1.09233 m
.94937 1.09233 L
.94937 1.11608 L
.93671 1.11608 L
F
.91139 1.09233 m
.92405 1.09233 L
.92405 1.11608 L
.91139 1.11608 L
F
.02532 1.11608 m
.03797 1.11608 L
.03797 1.13983 L
.02532 1.13983 L
F
.97468 1.09233 m
.98734 1.09233 L
.98734 1.11608 L
.97468 1.11608 L
F
.92405 1.09233 m
.93671 1.09233 L
.93671 1.11608 L
.92405 1.11608 L
F
.67089 1.13983 m
.68354 1.13983 L
.68354 1.16357 L
.67089 1.16357 L
F
.64557 1.13983 m
.65823 1.13983 L
.65823 1.16357 L
.64557 1.16357 L
F
.62025 1.13983 m
.63291 1.13983 L
.63291 1.16357 L
.62025 1.16357 L
F
.83544 1.13983 m
.8481 1.13983 L
.8481 1.16357 L
.83544 1.16357 L
F
.65823 1.13983 m
.67089 1.13983 L
.67089 1.16357 L
.65823 1.16357 L
F
.72152 1.13983 m
.73418 1.13983 L
.73418 1.16357 L
.72152 1.16357 L
F
.70886 1.13983 m
.72152 1.13983 L
.72152 1.16357 L
.70886 1.16357 L
F
.60759 1.13983 m
.62025 1.13983 L
.62025 1.16357 L
.60759 1.16357 L
F
.63291 1.13983 m
.64557 1.13983 L
.64557 1.16357 L
.63291 1.16357 L
F
.88608 .83112 m
.89873 .83112 L
.89873 .85487 L
.88608 .85487 L
F
.89873 .83112 m
.91139 .83112 L
.91139 .85487 L
.89873 .85487 L
F
.91139 .83112 m
.92405 .83112 L
.92405 .85487 L
.91139 .85487 L
F
.78481 .83112 m
.79747 .83112 L
.79747 .85487 L
.78481 .85487 L
F
.81013 .83112 m
.82278 .83112 L
.82278 .85487 L
.81013 .85487 L
F
.68354 .83112 m
.6962 .83112 L
.6962 .85487 L
.68354 .85487 L
F
.6962 .83112 m
.70886 .83112 L
.70886 .85487 L
.6962 .85487 L
F
.70886 .83112 m
.72152 .83112 L
.72152 .85487 L
.70886 .85487 L
F
.86076 .83112 m
.87342 .83112 L
.87342 .85487 L
.86076 .85487 L
F
.87342 .83112 m
.88608 .83112 L
.88608 .85487 L
.87342 .85487 L
F
.92405 .80738 m
.93671 .80738 L
.93671 .83112 L
.92405 .83112 L
F
.97468 .80738 m
.98734 .80738 L
.98734 .83112 L
.97468 .83112 L
F
.93671 .80738 m
.94937 .80738 L
.94937 .83112 L
.93671 .83112 L
F
.96203 .80738 m
.97468 .80738 L
.97468 .83112 L
.96203 .83112 L
F
.94937 .80738 m
.96203 .80738 L
.96203 .83112 L
.94937 .83112 L
F
.8481 .83112 m
.86076 .83112 L
.86076 .85487 L
.8481 .85487 L
F
.79747 .83112 m
.81013 .83112 L
.81013 .85487 L
.79747 .85487 L
F
.83544 .83112 m
.8481 .83112 L
.8481 .85487 L
.83544 .85487 L
F
.82278 .83112 m
.83544 .83112 L
.83544 .85487 L
.82278 .85487 L
F
0 1.13983 m
.01266 1.13983 L
.01266 1.16357 L
0 1.16357 L
F
.97468 1.11608 m
.98734 1.11608 L
.98734 1.13983 L
.97468 1.13983 L
F
.07595 1.13983 m
.08861 1.13983 L
.08861 1.16357 L
.07595 1.16357 L
F
.03797 1.13983 m
.05063 1.13983 L
.05063 1.16357 L
.03797 1.16357 L
F
.98734 1.11608 m
1 1.11608 L
1 1.13983 L
.98734 1.13983 L
F
.88608 1.11608 m
.89873 1.11608 L
.89873 1.13983 L
.88608 1.13983 L
F
.06329 1.13983 m
.07595 1.13983 L
.07595 1.16357 L
.06329 1.16357 L
F
.87342 1.11608 m
.88608 1.11608 L
.88608 1.13983 L
.87342 1.13983 L
F
.59904 1.13983 m
.60759 1.13983 L
.60759 1.16357 L
.59888 1.16357 L
F
.05063 1.13983 m
.06329 1.13983 L
.06329 1.16357 L
.05063 1.16357 L
F
.74684 .83112 m
.75949 .83112 L
.75949 .85487 L
.74684 .85487 L
F
.75949 .83112 m
.77215 .83112 L
.77215 .85487 L
.75949 .85487 L
F
.77215 .83112 m
.78481 .83112 L
.78481 .85487 L
.77215 .85487 L
F
.65823 .83112 m
.67089 .83112 L
.67089 .85487 L
.65823 .85487 L
F
.67089 .83112 m
.68354 .83112 L
.68354 .85487 L
.67089 .85487 L
F
.02532 1.13983 m
.03797 1.13983 L
.03797 1.16357 L
.02532 1.16357 L
F
.01266 1.13983 m
.02532 1.13983 L
.02532 1.16357 L
.01266 1.16357 L
F
.72152 .83112 m
.73418 .83112 L
.73418 .85487 L
.72152 .85487 L
F
.73418 .83112 m
.74684 .83112 L
.74684 .85487 L
.73418 .85487 L
F
.91139 1.16357 m
.92405 1.16357 L
.92405 1.18732 L
.91139 1.18732 L
F
.92405 1.16357 m
.93671 1.16357 L
.93671 1.18732 L
.92405 1.18732 L
F
.93671 1.16357 m
.94937 1.16357 L
.94937 1.18732 L
.93671 1.18732 L
F
.94937 1.16357 m
.96203 1.16357 L
.96203 1.18732 L
.94937 1.18732 L
F
.97468 1.16357 m
.98734 1.16357 L
.98734 1.18732 L
.97468 1.18732 L
F
.98734 1.16357 m
1 1.16357 L
1 1.18732 L
.98734 1.18732 L
F
.03797 1.18732 m
.05063 1.18732 L
.05063 1.21107 L
.03797 1.21107 L
F
0 1.18732 m
.01266 1.18732 L
.01266 1.21107 L
0 1.21107 L
F
.02532 1.18732 m
.03797 1.18732 L
.03797 1.21107 L
.02532 1.21107 L
F
.01266 1.18732 m
.02532 1.18732 L
.02532 1.21107 L
.01266 1.21107 L
F
.63291 1.18732 m
.64557 1.18732 L
.64557 1.21107 L
.63291 1.21107 L
F
.07595 1.18732 m
.08861 1.18732 L
.08861 1.21107 L
.07595 1.21107 L
F
.62025 1.18732 m
.63291 1.18732 L
.63291 1.21107 L
.62025 1.21107 L
F
.06329 1.18732 m
.07595 1.18732 L
.07595 1.21107 L
.06329 1.21107 L
F
.05063 1.18732 m
.06329 1.18732 L
.06329 1.21107 L
.05063 1.21107 L
F
.60759 1.18732 m
.62025 1.18732 L
.62025 1.21107 L
.60759 1.21107 L
F
.96203 1.16357 m
.97468 1.16357 L
.97468 1.18732 L
.96203 1.18732 L
F
.64557 1.18732 m
.65823 1.18732 L
.65823 1.21107 L
.64557 1.21107 L
F
.59854 1.18732 m
.60759 1.18732 L
.60759 1.21107 L
.59786 1.21107 L
F
.6962 1.18732 m
.70886 1.18732 L
.70886 1.21107 L
.6962 1.21107 L
F
.70886 1.18732 m
.72152 1.18732 L
.72152 1.21107 L
.70886 1.21107 L
F
.72152 1.18732 m
.73418 1.18732 L
.73418 1.21107 L
.72152 1.21107 L
F
.87342 1.18732 m
.88608 1.18732 L
.88608 1.21107 L
.87342 1.21107 L
F
.88608 1.18732 m
.89873 1.18732 L
.89873 1.21107 L
.88608 1.21107 L
F
.77215 1.18732 m
.78481 1.18732 L
.78481 1.21107 L
.77215 1.21107 L
F
.68354 1.18732 m
.6962 1.18732 L
.6962 1.21107 L
.68354 1.21107 L
F
.75949 1.18732 m
.77215 1.18732 L
.77215 1.21107 L
.75949 1.21107 L
F
.67089 1.18732 m
.68354 1.18732 L
.68354 1.21107 L
.67089 1.21107 L
F
.65823 1.18732 m
.67089 1.18732 L
.67089 1.21107 L
.65823 1.21107 L
F
.81013 1.18732 m
.82278 1.18732 L
.82278 1.21107 L
.81013 1.21107 L
F
.86076 1.18732 m
.87342 1.18732 L
.87342 1.21107 L
.86076 1.21107 L
F
.79747 1.18732 m
.81013 1.18732 L
.81013 1.21107 L
.79747 1.21107 L
F
.8481 1.18732 m
.86076 1.18732 L
.86076 1.21107 L
.8481 1.21107 L
F
.83544 1.18732 m
.8481 1.18732 L
.8481 1.21107 L
.83544 1.21107 L
F
.89873 1.18732 m
.91139 1.18732 L
.91139 1.21107 L
.89873 1.21107 L
F
.92405 1.18732 m
.93671 1.18732 L
.93671 1.21107 L
.92405 1.21107 L
F
.82278 1.18732 m
.83544 1.18732 L
.83544 1.21107 L
.82278 1.21107 L
F
.91139 1.18732 m
.92405 1.18732 L
.92405 1.21107 L
.91139 1.21107 L
F
.89873 1.09233 m
.91139 1.09233 L
.91139 1.11608 L
.89873 1.11608 L
F
.87342 1.09233 m
.88608 1.09233 L
.88608 1.11608 L
.87342 1.11608 L
F
.8481 1.09233 m
.86076 1.09233 L
.86076 1.11608 L
.8481 1.11608 L
F
.79747 1.09233 m
.81013 1.09233 L
.81013 1.11608 L
.79747 1.11608 L
F
.88608 1.09233 m
.89873 1.09233 L
.89873 1.11608 L
.88608 1.11608 L
F
.75949 1.11608 m
.77215 1.11608 L
.77215 1.13983 L
.75949 1.13983 L
F
.73418 1.11608 m
.74684 1.11608 L
.74684 1.13983 L
.73418 1.13983 L
F
.70886 1.11608 m
.72152 1.11608 L
.72152 1.13983 L
.70886 1.13983 L
F
.86076 1.09233 m
.87342 1.09233 L
.87342 1.11608 L
.86076 1.11608 L
F
.74684 1.11608 m
.75949 1.11608 L
.75949 1.13983 L
.74684 1.13983 L
F
0 1.11608 m
.01266 1.11608 L
.01266 1.13983 L
0 1.13983 L
F
.98734 1.09233 m
1 1.09233 L
1 1.11608 L
.98734 1.11608 L
F
.82278 1.09233 m
.83544 1.09233 L
.83544 1.11608 L
.82278 1.11608 L
F
.01266 1.11608 m
.02532 1.11608 L
.02532 1.13983 L
.01266 1.13983 L
F
.03797 1.11608 m
.05063 1.11608 L
.05063 1.13983 L
.03797 1.13983 L
F
.78481 1.09233 m
.79747 1.09233 L
.79747 1.11608 L
.78481 1.11608 L
F
.77215 1.09233 m
.78481 1.09233 L
.78481 1.11608 L
.77215 1.11608 L
F
.81013 1.09233 m
.82278 1.09233 L
.82278 1.11608 L
.81013 1.11608 L
F
.83544 1.09233 m
.8481 1.09233 L
.8481 1.11608 L
.83544 1.11608 L
F
.67089 1.11608 m
.68354 1.11608 L
.68354 1.13983 L
.67089 1.13983 L
F
.05063 1.11608 m
.06329 1.11608 L
.06329 1.13983 L
.05063 1.13983 L
F
.65823 1.11608 m
.67089 1.11608 L
.67089 1.13983 L
.65823 1.13983 L
F
.62025 1.11608 m
.63291 1.11608 L
.63291 1.13983 L
.62025 1.13983 L
F
.06329 1.11608 m
.07595 1.11608 L
.07595 1.13983 L
.06329 1.13983 L
F
.10127 1.18732 m
.11392 1.18732 L
.11392 1.21107 L
.10127 1.21107 L
F
.64557 1.11608 m
.65823 1.11608 L
.65823 1.13983 L
.64557 1.13983 L
F
.08861 1.18732 m
.10127 1.18732 L
.10127 1.21107 L
.08861 1.21107 L
F
.68354 1.11608 m
.6962 1.11608 L
.6962 1.13983 L
.68354 1.13983 L
F
.63291 1.11608 m
.64557 1.11608 L
.64557 1.13983 L
.63291 1.13983 L
F
.79747 1.11608 m
.81013 1.11608 L
.81013 1.13983 L
.79747 1.13983 L
F
.81013 1.11608 m
.82278 1.11608 L
.82278 1.13983 L
.81013 1.13983 L
F
.82278 1.11608 m
.83544 1.11608 L
.83544 1.13983 L
.82278 1.13983 L
F
.6962 1.11608 m
.70886 1.11608 L
.70886 1.13983 L
.6962 1.13983 L
F
.72152 1.11608 m
.73418 1.11608 L
.73418 1.13983 L
.72152 1.13983 L
F
.60759 1.11608 m
.62025 1.11608 L
.62025 1.13983 L
.60759 1.13983 L
F
.5991 1.11608 m
.60759 1.11608 L
.60759 1.13983 L
.59904 1.13983 L
F
.77215 1.11608 m
.78481 1.11608 L
.78481 1.13983 L
.77215 1.13983 L
F
.78481 1.11608 m
.79747 1.11608 L
.79747 1.13983 L
.78481 1.13983 L
F
.88608 .75989 m
.89873 .75989 L
.89873 .78363 L
.88608 .78363 L
F
.86076 .75989 m
.87342 .75989 L
.87342 .78363 L
.86076 .78363 L
F
.83544 .75989 m
.8481 .75989 L
.8481 .78363 L
.83544 .78363 L
F
.65823 .80738 m
.67089 .80738 L
.67089 .83112 L
.65823 .83112 L
F
.87342 .75989 m
.88608 .75989 L
.88608 .78363 L
.87342 .78363 L
F
.92405 .75989 m
.93671 .75989 L
.93671 .78363 L
.92405 .78363 L
F
.93671 .75989 m
.94937 .75989 L
.94937 .78363 L
.93671 .78363 L
F
.94937 .75989 m
.96203 .75989 L
.96203 .78363 L
.94937 .78363 L
F
.82278 .75989 m
.83544 .75989 L
.83544 .78363 L
.82278 .78363 L
F
.8481 .75989 m
.86076 .75989 L
.86076 .78363 L
.8481 .78363 L
F
.03797 .80738 m
.05063 .80738 L
.05063 .83112 L
.03797 .83112 L
F
.05063 .80738 m
.06329 .80738 L
.06329 .83112 L
.05063 .83112 L
F
.59911 .80738 m
.60759 .80738 L
.60759 .83112 L
.59911 .83112 L
F
.60759 .80738 m
.62025 .80738 L
.62025 .83112 L
.60759 .83112 L
F
.63291 .80738 m
.64557 .80738 L
.64557 .83112 L
.63291 .83112 L
F
.6962 .80738 m
.70886 .80738 L
.70886 .83112 L
.6962 .83112 L
F
.64557 .80738 m
.65823 .80738 L
.65823 .83112 L
.64557 .83112 L
F
.68354 .80738 m
.6962 .80738 L
.6962 .83112 L
.68354 .83112 L
F
.67089 .80738 m
.68354 .80738 L
.68354 .83112 L
.67089 .83112 L
F
.77215 .75989 m
.78481 .75989 L
.78481 .78363 L
.77215 .78363 L
F
.75949 .75989 m
.77215 .75989 L
.77215 .78363 L
.75949 .78363 L
F
.67089 .78363 m
.68354 .78363 L
.68354 .80738 L
.67089 .80738 L
F
.78481 .75989 m
.79747 .75989 L
.79747 .78363 L
.78481 .78363 L
F
.81013 .75989 m
.82278 .75989 L
.82278 .78363 L
.81013 .78363 L
F
.62025 .78363 m
.63291 .78363 L
.63291 .80738 L
.62025 .80738 L
F
.63291 .78363 m
.64557 .78363 L
.64557 .80738 L
.63291 .80738 L
F
.64557 .78363 m
.65823 .78363 L
.65823 .80738 L
.64557 .80738 L
F
.65823 .78363 m
.67089 .78363 L
.67089 .80738 L
.65823 .80738 L
F
.68354 .78363 m
.6962 .78363 L
.6962 .80738 L
.68354 .80738 L
F
.72152 .75989 m
.73418 .75989 L
.73418 .78363 L
.72152 .78363 L
F
.73418 .75989 m
.74684 .75989 L
.74684 .78363 L
.73418 .78363 L
F
.74684 .75989 m
.75949 .75989 L
.75949 .78363 L
.74684 .78363 L
F
.89873 .75989 m
.91139 .75989 L
.91139 .78363 L
.89873 .78363 L
F
.91139 .75989 m
.92405 .75989 L
.92405 .78363 L
.91139 .78363 L
F
.70886 .75989 m
.72152 .75989 L
.72152 .78363 L
.70886 .78363 L
F
.79747 .75989 m
.81013 .75989 L
.81013 .78363 L
.79747 .78363 L
F
.6962 .75989 m
.70886 .75989 L
.70886 .78363 L
.6962 .78363 L
F
.68354 .75989 m
.6962 .75989 L
.6962 .78363 L
.68354 .78363 L
F
.02532 .80738 m
.03797 .80738 L
.03797 .83112 L
.02532 .83112 L
F
0 .80738 m
.01266 .80738 L
.01266 .83112 L
0 .83112 L
F
.97468 .78363 m
.98734 .78363 L
.98734 .80738 L
.97468 .80738 L
F
.92405 .78363 m
.93671 .78363 L
.93671 .80738 L
.92405 .80738 L
F
.01266 .80738 m
.02532 .80738 L
.02532 .83112 L
.01266 .83112 L
F
.82278 .78363 m
.83544 .78363 L
.83544 .80738 L
.82278 .80738 L
F
.79747 .78363 m
.81013 .78363 L
.81013 .80738 L
.79747 .80738 L
F
.77215 .78363 m
.78481 .78363 L
.78481 .80738 L
.77215 .80738 L
F
.98734 .78363 m
1 .78363 L
1 .80738 L
.98734 .80738 L
F
.81013 .78363 m
.82278 .78363 L
.82278 .80738 L
.81013 .80738 L
F
.65823 .94986 m
.67089 .94986 L
.67089 .9736 L
.65823 .9736 L
F
.64557 .94986 m
.65823 .94986 L
.65823 .9736 L
.64557 .9736 L
F
.94937 .78363 m
.96203 .78363 L
.96203 .80738 L
.94937 .80738 L
F
.67089 .94986 m
.68354 .94986 L
.68354 .9736 L
.67089 .9736 L
F
.6962 .94986 m
.70886 .94986 L
.70886 .9736 L
.6962 .9736 L
F
.91139 .78363 m
.92405 .78363 L
.92405 .80738 L
.91139 .80738 L
F
.89873 .78363 m
.91139 .78363 L
.91139 .80738 L
.89873 .80738 L
F
.93671 .78363 m
.94937 .78363 L
.94937 .80738 L
.93671 .80738 L
F
.96203 .78363 m
.97468 .78363 L
.97468 .80738 L
.96203 .80738 L
F
.73418 .80738 m
.74684 .80738 L
.74684 .83112 L
.73418 .83112 L
F
.70886 .80738 m
.72152 .80738 L
.72152 .83112 L
.70886 .83112 L
F
.82278 .80738 m
.83544 .80738 L
.83544 .83112 L
.82278 .83112 L
F
.77215 .80738 m
.78481 .80738 L
.78481 .83112 L
.77215 .83112 L
F
.72152 .80738 m
.73418 .80738 L
.73418 .83112 L
.72152 .83112 L
F
.79747 .80738 m
.81013 .80738 L
.81013 .83112 L
.79747 .83112 L
F
.78481 .80738 m
.79747 .80738 L
.79747 .83112 L
.78481 .83112 L
F
.62025 .80738 m
.63291 .80738 L
.63291 .83112 L
.62025 .83112 L
F
.81013 .80738 m
.82278 .80738 L
.82278 .83112 L
.81013 .83112 L
F
.83544 .80738 m
.8481 .80738 L
.8481 .83112 L
.83544 .83112 L
F
.86076 .78363 m
.87342 .78363 L
.87342 .80738 L
.86076 .80738 L
F
.87342 .78363 m
.88608 .78363 L
.88608 .80738 L
.87342 .80738 L
F
.88608 .78363 m
.89873 .78363 L
.89873 .80738 L
.88608 .80738 L
F
.75949 .78363 m
.77215 .78363 L
.77215 .80738 L
.75949 .80738 L
F
.78481 .78363 m
.79747 .78363 L
.79747 .80738 L
.78481 .80738 L
F
.75949 .80738 m
.77215 .80738 L
.77215 .83112 L
.75949 .83112 L
F
.74684 .80738 m
.75949 .80738 L
.75949 .83112 L
.74684 .83112 L
F
.83544 .78363 m
.8481 .78363 L
.8481 .80738 L
.83544 .80738 L
F
.8481 .78363 m
.86076 .78363 L
.86076 .80738 L
.8481 .80738 L
F
.79747 .85487 m
.81013 .85487 L
.81013 .87862 L
.79747 .87862 L
F
.77215 .85487 m
.78481 .85487 L
.78481 .87862 L
.77215 .87862 L
F
.74684 .85487 m
.75949 .85487 L
.75949 .87862 L
.74684 .87862 L
F
.96203 .85487 m
.97468 .85487 L
.97468 .87862 L
.96203 .87862 L
F
.78481 .85487 m
.79747 .85487 L
.79747 .87862 L
.78481 .87862 L
F
.83544 .85487 m
.8481 .85487 L
.8481 .87862 L
.83544 .87862 L
F
.8481 .85487 m
.86076 .85487 L
.86076 .87862 L
.8481 .87862 L
F
.86076 .85487 m
.87342 .85487 L
.87342 .87862 L
.86076 .87862 L
F
.73418 .85487 m
.74684 .85487 L
.74684 .87862 L
.73418 .87862 L
F
.75949 .85487 m
.77215 .85487 L
.77215 .87862 L
.75949 .87862 L
F
.87342 .85487 m
.88608 .85487 L
.88608 .87862 L
.87342 .87862 L
F
.88608 .85487 m
.89873 .85487 L
.89873 .87862 L
.88608 .87862 L
F
.89873 .85487 m
.91139 .85487 L
.91139 .87862 L
.89873 .87862 L
F
.91139 .85487 m
.92405 .85487 L
.92405 .87862 L
.91139 .87862 L
F
.93671 .85487 m
.94937 .85487 L
.94937 .87862 L
.93671 .87862 L
F
0 .87862 m
.01266 .87862 L
.01266 .90236 L
0 .90236 L
F
.94937 .85487 m
.96203 .85487 L
.96203 .87862 L
.94937 .87862 L
F
.98734 .85487 m
1 .85487 L
1 .87862 L
.98734 .87862 L
F
.97468 .85487 m
.98734 .85487 L
.98734 .87862 L
.97468 .87862 L
F
.60759 .83112 m
.62025 .83112 L
.62025 .85487 L
.60759 .85487 L
F
.59911 .83112 m
.60759 .83112 L
.60759 .85487 L
.59911 .85487 L
F
.89873 .80738 m
.91139 .80738 L
.91139 .83112 L
.89873 .83112 L
F
.62025 .83112 m
.63291 .83112 L
.63291 .85487 L
.62025 .85487 L
F
.64557 .83112 m
.65823 .83112 L
.65823 .85487 L
.64557 .85487 L
F
.8481 .80738 m
.86076 .80738 L
.86076 .83112 L
.8481 .83112 L
F
.86076 .80738 m
.87342 .80738 L
.87342 .83112 L
.86076 .83112 L
F
.87342 .80738 m
.88608 .80738 L
.88608 .83112 L
.87342 .83112 L
F
.88608 .80738 m
.89873 .80738 L
.89873 .83112 L
.88608 .83112 L
F
.91139 .80738 m
.92405 .80738 L
.92405 .83112 L
.91139 .83112 L
F
.02532 .83112 m
.03797 .83112 L
.03797 .85487 L
.02532 .85487 L
F
.03797 .83112 m
.05063 .83112 L
.05063 .85487 L
.03797 .85487 L
F
.05063 .83112 m
.06329 .83112 L
.06329 .85487 L
.05063 .85487 L
F
.81013 .85487 m
.82278 .85487 L
.82278 .87862 L
.81013 .87862 L
F
.82278 .85487 m
.83544 .85487 L
.83544 .87862 L
.82278 .87862 L
F
.01266 .83112 m
.02532 .83112 L
.02532 .85487 L
.01266 .85487 L
F
.63291 .83112 m
.64557 .83112 L
.64557 .85487 L
.63291 .85487 L
F
0 .83112 m
.01266 .83112 L
.01266 .85487 L
0 .85487 L
F
.98734 .80738 m
1 .80738 L
1 .83112 L
.98734 .83112 L
F
.60759 .78363 m
.62025 .78363 L
.62025 .80738 L
.60759 .80738 L
F
.05063 .78363 m
.06329 .78363 L
.06329 .80738 L
.05063 .80738 L
F
.02532 .78363 m
.03797 .78363 L
.03797 .80738 L
.02532 .80738 L
F
.97468 .75989 m
.98734 .75989 L
.98734 .78363 L
.97468 .78363 L
F
.59911 .78363 m
.60759 .78363 L
.60759 .80738 L
.59911 .80738 L
F
.65823 .85487 m
.67089 .85487 L
.67089 .87862 L
.65823 .87862 L
F
.63291 .85487 m
.64557 .85487 L
.64557 .87862 L
.63291 .87862 L
F
.60759 .85487 m
.62025 .85487 L
.62025 .87862 L
.60759 .87862 L
F
.03797 .78363 m
.05063 .78363 L
.05063 .80738 L
.03797 .80738 L
F
.64557 .85487 m
.65823 .85487 L
.65823 .87862 L
.64557 .87862 L
F
.6962 .78363 m
.70886 .78363 L
.70886 .80738 L
.6962 .80738 L
F
.74684 .78363 m
.75949 .78363 L
.75949 .80738 L
.74684 .80738 L
F
.70886 .78363 m
.72152 .78363 L
.72152 .80738 L
.70886 .80738 L
F
.73418 .78363 m
.74684 .78363 L
.74684 .80738 L
.73418 .80738 L
F
.72152 .78363 m
.73418 .78363 L
.73418 .80738 L
.72152 .80738 L
F
.01266 .78363 m
.02532 .78363 L
.02532 .80738 L
.01266 .80738 L
F
.96203 .75989 m
.97468 .75989 L
.97468 .78363 L
.96203 .78363 L
F
0 .78363 m
.01266 .78363 L
.01266 .80738 L
0 .80738 L
F
.98734 .75989 m
1 .75989 L
1 .78363 L
.98734 .78363 L
F
.94937 .83112 m
.96203 .83112 L
.96203 .85487 L
.94937 .85487 L
F
.92405 .83112 m
.93671 .83112 L
.93671 .85487 L
.92405 .85487 L
F
.03797 .85487 m
.05063 .85487 L
.05063 .87862 L
.03797 .87862 L
F
.98734 .83112 m
1 .83112 L
1 .85487 L
.98734 .85487 L
F
.93671 .83112 m
.94937 .83112 L
.94937 .85487 L
.93671 .85487 L
F
.01266 .85487 m
.02532 .85487 L
.02532 .87862 L
.01266 .87862 L
F
0 .85487 m
.01266 .85487 L
.01266 .87862 L
0 .87862 L
F
.92405 .85487 m
.93671 .85487 L
.93671 .87862 L
.92405 .87862 L
F
.02532 .85487 m
.03797 .85487 L
.03797 .87862 L
.02532 .87862 L
F
.05063 .85487 m
.06329 .85487 L
.06329 .87862 L
.05063 .87862 L
F
.6962 .85487 m
.70886 .85487 L
.70886 .87862 L
.6962 .87862 L
F
.70886 .85487 m
.72152 .85487 L
.72152 .87862 L
.70886 .87862 L
F
.72152 .85487 m
.73418 .85487 L
.73418 .87862 L
.72152 .87862 L
F
.59911 .85487 m
.60759 .85487 L
.60759 .87862 L
.59911 .87862 L
F
.62025 .85487 m
.63291 .85487 L
.63291 .87862 L
.62025 .87862 L
F
.97468 .83112 m
.98734 .83112 L
.98734 .85487 L
.97468 .85487 L
F
.96203 .83112 m
.97468 .83112 L
.97468 .85487 L
.96203 .85487 L
F
.67089 .85487 m
.68354 .85487 L
.68354 .87862 L
.67089 .87862 L
F
.68354 .85487 m
.6962 .85487 L
.6962 .87862 L
.68354 .87862 L
F
.05063 1.06859 m
.06329 1.06859 L
.06329 1.09233 L
.05063 1.09233 L
F
.62025 1.06859 m
.63291 1.06859 L
.63291 1.09233 L
.62025 1.09233 L
F
.03797 1.06859 m
.05063 1.06859 L
.05063 1.09233 L
.03797 1.09233 L
F
.60759 1.06859 m
.62025 1.06859 L
.62025 1.09233 L
.60759 1.09233 L
F
.59911 1.06859 m
.60759 1.06859 L
.60759 1.09233 L
.59911 1.09233 L
F
.63291 1.06859 m
.64557 1.06859 L
.64557 1.09233 L
.63291 1.09233 L
F
.68354 1.06859 m
.6962 1.06859 L
.6962 1.09233 L
.68354 1.09233 L
F
.64557 1.06859 m
.65823 1.06859 L
.65823 1.09233 L
.64557 1.09233 L
F
.67089 1.06859 m
.68354 1.06859 L
.68354 1.09233 L
.67089 1.09233 L
F
.65823 1.06859 m
.67089 1.06859 L
.67089 1.09233 L
.65823 1.09233 L
F
.65823 1.04484 m
.67089 1.04484 L
.67089 1.06859 L
.65823 1.06859 L
F
.63291 1.04484 m
.64557 1.04484 L
.64557 1.06859 L
.63291 1.06859 L
F
.74684 1.04484 m
.75949 1.04484 L
.75949 1.06859 L
.74684 1.06859 L
F
.6962 1.04484 m
.70886 1.04484 L
.70886 1.06859 L
.6962 1.06859 L
F
.64557 1.04484 m
.65823 1.04484 L
.65823 1.06859 L
.64557 1.06859 L
F
.70886 1.04484 m
.72152 1.04484 L
.72152 1.06859 L
.70886 1.06859 L
F
.72152 1.04484 m
.73418 1.04484 L
.73418 1.06859 L
.72152 1.06859 L
F
.73418 1.04484 m
.74684 1.04484 L
.74684 1.06859 L
.73418 1.06859 L
F
.75949 1.04484 m
.77215 1.04484 L
.77215 1.06859 L
.75949 1.06859 L
F
.8481 .49867 m
.86076 .49867 L
.86076 .52242 L
.8481 .52242 L
F
.82278 .49867 m
.83544 .49867 L
.83544 .52242 L
.82278 .52242 L
F
.79747 .49867 m
.81013 .49867 L
.81013 .52242 L
.79747 .52242 L
F
.98734 1.04484 m
1 1.04484 L
1 1.06859 L
.98734 1.06859 L
F
.83544 .49867 m
.8481 .49867 L
.8481 .52242 L
.83544 .52242 L
F
.88608 .49867 m
.89873 .49867 L
.89873 .52242 L
.88608 .52242 L
F
.89873 .49867 m
.91139 .49867 L
.91139 .52242 L
.89873 .52242 L
F
.91139 .49867 m
.92405 .49867 L
.92405 .52242 L
.91139 .52242 L
F
.78481 .49867 m
.79747 .49867 L
.79747 .52242 L
.78481 .52242 L
F
.81013 .49867 m
.82278 .49867 L
.82278 .52242 L
.81013 .52242 L
F
.91139 1.04484 m
.92405 1.04484 L
.92405 1.06859 L
.91139 1.06859 L
F
.96203 1.04484 m
.97468 1.04484 L
.97468 1.06859 L
.96203 1.06859 L
F
.92405 1.04484 m
.93671 1.04484 L
.93671 1.06859 L
.92405 1.06859 L
F
.94937 1.04484 m
.96203 1.04484 L
.96203 1.06859 L
.94937 1.06859 L
F
.93671 1.04484 m
.94937 1.04484 L
.94937 1.06859 L
.93671 1.06859 L
F
.02532 1.06859 m
.03797 1.06859 L
.03797 1.09233 L
.02532 1.09233 L
F
.97468 1.04484 m
.98734 1.04484 L
.98734 1.06859 L
.97468 1.06859 L
F
.01266 1.06859 m
.02532 1.06859 L
.02532 1.09233 L
.01266 1.09233 L
F
0 1.06859 m
.01266 1.06859 L
.01266 1.09233 L
0 1.09233 L
F
.70886 1.09233 m
.72152 1.09233 L
.72152 1.11608 L
.70886 1.11608 L
F
.75949 1.09233 m
.77215 1.09233 L
.77215 1.11608 L
.75949 1.11608 L
F
.72152 1.09233 m
.73418 1.09233 L
.73418 1.11608 L
.72152 1.11608 L
F
.74684 1.09233 m
.75949 1.09233 L
.75949 1.11608 L
.74684 1.11608 L
F
.73418 1.09233 m
.74684 1.09233 L
.74684 1.11608 L
.73418 1.11608 L
F
.98734 1.06859 m
1 1.06859 L
1 1.09233 L
.98734 1.09233 L
F
.03797 1.09233 m
.05063 1.09233 L
.05063 1.11608 L
.03797 1.11608 L
F
.97468 1.06859 m
.98734 1.06859 L
.98734 1.09233 L
.97468 1.09233 L
F
.02532 1.09233 m
.03797 1.09233 L
.03797 1.11608 L
.02532 1.11608 L
F
.01266 1.09233 m
.02532 1.09233 L
.02532 1.11608 L
.01266 1.11608 L
F
.68354 1.09233 m
.6962 1.09233 L
.6962 1.11608 L
.68354 1.11608 L
F
.78481 1.06859 m
.79747 1.06859 L
.79747 1.09233 L
.78481 1.09233 L
F
.67089 1.09233 m
.68354 1.09233 L
.68354 1.11608 L
.67089 1.11608 L
F
.82278 1.06859 m
.83544 1.06859 L
.83544 1.09233 L
.82278 1.09233 L
F
.77215 1.06859 m
.78481 1.06859 L
.78481 1.09233 L
.77215 1.09233 L
F
.63291 1.09233 m
.64557 1.09233 L
.64557 1.11608 L
.63291 1.11608 L
F
.65823 1.09233 m
.67089 1.09233 L
.67089 1.11608 L
.65823 1.11608 L
F
.6962 1.09233 m
.70886 1.09233 L
.70886 1.11608 L
.6962 1.11608 L
F
.64557 1.09233 m
.65823 1.09233 L
.65823 1.11608 L
.64557 1.11608 L
F
.88608 1.04484 m
.89873 1.04484 L
.89873 1.06859 L
.88608 1.06859 L
F
.79747 1.04484 m
.81013 1.04484 L
.81013 1.06859 L
.79747 1.06859 L
F
.87342 1.04484 m
.88608 1.04484 L
.88608 1.06859 L
.87342 1.06859 L
F
.78481 1.04484 m
.79747 1.04484 L
.79747 1.06859 L
.78481 1.06859 L
F
.77215 1.04484 m
.78481 1.04484 L
.78481 1.06859 L
.77215 1.06859 L
F
.68354 1.04484 m
.6962 1.04484 L
.6962 1.06859 L
.68354 1.06859 L
F
.86076 1.04484 m
.87342 1.04484 L
.87342 1.06859 L
.86076 1.06859 L
F
.67089 1.04484 m
.68354 1.04484 L
.68354 1.06859 L
.67089 1.06859 L
F
.89873 1.04484 m
.91139 1.04484 L
.91139 1.06859 L
.89873 1.06859 L
F
.8481 1.04484 m
.86076 1.04484 L
.86076 1.06859 L
.8481 1.06859 L
F
.62025 1.09233 m
.63291 1.09233 L
.63291 1.11608 L
.62025 1.11608 L
F
.59911 1.09233 m
.60759 1.09233 L
.60759 1.11608 L
.5991 1.11608 L
F
.05063 1.09233 m
.06329 1.09233 L
.06329 1.11608 L
.05063 1.11608 L
F
0 1.09233 m
.01266 1.09233 L
.01266 1.11608 L
0 1.11608 L
F
.60759 1.09233 m
.62025 1.09233 L
.62025 1.11608 L
.60759 1.11608 L
F
.81013 1.04484 m
.82278 1.04484 L
.82278 1.06859 L
.81013 1.06859 L
F
.83544 1.04484 m
.8481 1.04484 L
.8481 1.06859 L
.83544 1.06859 L
F
.06329 1.09233 m
.07595 1.09233 L
.07595 1.11608 L
.06329 1.11608 L
F
.82278 1.04484 m
.83544 1.04484 L
.83544 1.06859 L
.82278 1.06859 L
F
.77215 .47493 m
.78481 .47493 L
.78481 .49867 L
.77215 .49867 L
F
.78481 .47493 m
.79747 .47493 L
.79747 .49867 L
.78481 .49867 L
F
.79747 .47493 m
.81013 .47493 L
.81013 .49867 L
.79747 .49867 L
F
.81013 .47493 m
.82278 .47493 L
.82278 .49867 L
.81013 .49867 L
F
.83544 .47493 m
.8481 .47493 L
.8481 .49867 L
.83544 .49867 L
F
.8481 .47493 m
.86076 .47493 L
.86076 .49867 L
.8481 .49867 L
F
.89873 .47493 m
.91139 .47493 L
.91139 .49867 L
.89873 .49867 L
F
.86076 .47493 m
.87342 .47493 L
.87342 .49867 L
.86076 .49867 L
F
.88608 .47493 m
.89873 .47493 L
.89873 .49867 L
.88608 .49867 L
F
.87342 .47493 m
.88608 .47493 L
.88608 .49867 L
.87342 .49867 L
F
.02532 .49867 m
.03797 .49867 L
.03797 .52242 L
.02532 .52242 L
F
.93671 .47493 m
.94937 .47493 L
.94937 .49867 L
.93671 .49867 L
F
.01266 .49867 m
.02532 .49867 L
.02532 .52242 L
.01266 .52242 L
F
.92405 .47493 m
.93671 .47493 L
.93671 .49867 L
.92405 .49867 L
F
.91139 .47493 m
.92405 .47493 L
.92405 .49867 L
.91139 .49867 L
F
0 .49867 m
.01266 .49867 L
.01266 .52242 L
0 .52242 L
F
.82278 .47493 m
.83544 .47493 L
.83544 .49867 L
.82278 .49867 L
F
.03797 .49867 m
.05063 .49867 L
.05063 .52242 L
.03797 .52242 L
F
.98734 .47493 m
1 .47493 L
1 .49867 L
.98734 .49867 L
F
.11392 .49867 m
.12658 .49867 L
.12658 .52242 L
.11392 .52242 L
F
.08861 .49867 m
.10127 .49867 L
.10127 .52242 L
.08861 .52242 L
F
.06329 .49867 m
.07595 .49867 L
.07595 .52242 L
.06329 .52242 L
F
.59494 .49867 m
.60759 .49867 L
.60759 .52242 L
.59494 .52242 L
F
.10127 .49867 m
.11392 .49867 L
.11392 .52242 L
.10127 .52242 L
F
.1519 .49867 m
.16456 .49867 L
.16456 .52242 L
.1519 .52242 L
F
.16456 .49867 m
.17722 .49867 L
.17722 .52242 L
.16456 .52242 L
F
.17722 .49867 m
.18987 .49867 L
.18987 .52242 L
.17722 .52242 L
F
.05063 .49867 m
.06329 .49867 L
.06329 .52242 L
.05063 .52242 L
F
.07595 .49867 m
.08861 .49867 L
.08861 .52242 L
.07595 .52242 L
F
.18987 .49867 m
.20253 .49867 L
.20253 .52242 L
.18987 .52242 L
F
.20253 .49867 m
.21519 .49867 L
.21519 .52242 L
.20253 .52242 L
F
.21519 .49867 m
.22785 .49867 L
.22785 .52242 L
.21519 .52242 L
F
.22785 .49867 m
.24051 .49867 L
.24051 .52242 L
.22785 .52242 L
F
.56962 .49867 m
.58228 .49867 L
.58228 .52242 L
.56962 .52242 L
F
.63291 .49867 m
.64557 .49867 L
.64557 .52242 L
.63291 .52242 L
F
.58228 .49867 m
.59494 .49867 L
.59494 .52242 L
.58228 .52242 L
F
.62025 .49867 m
.63291 .49867 L
.63291 .52242 L
.62025 .52242 L
F
.60759 .49867 m
.62025 .49867 L
.62025 .52242 L
.60759 .52242 L
F
.73418 .49867 m
.74684 .49867 L
.74684 .52242 L
.73418 .52242 L
F
.72152 .49867 m
.73418 .49867 L
.73418 .52242 L
.72152 .52242 L
F
.11392 .52242 m
.12658 .52242 L
.12658 .54617 L
.11392 .54617 L
F
.74684 .49867 m
.75949 .49867 L
.75949 .52242 L
.74684 .52242 L
F
.77215 .49867 m
.78481 .49867 L
.78481 .52242 L
.77215 .52242 L
F
.06329 .52242 m
.07595 .52242 L
.07595 .54617 L
.06329 .54617 L
F
.07595 .52242 m
.08861 .52242 L
.08861 .54617 L
.07595 .54617 L
F
.08861 .52242 m
.10127 .52242 L
.10127 .54617 L
.08861 .54617 L
F
.10127 .52242 m
.11392 .52242 L
.11392 .54617 L
.10127 .54617 L
F
.12658 .52242 m
.13924 .52242 L
.13924 .54617 L
.12658 .54617 L
F
.68354 .49867 m
.6962 .49867 L
.6962 .52242 L
.68354 .52242 L
F
.6962 .49867 m
.70886 .49867 L
.70886 .52242 L
.6962 .52242 L
F
.70886 .49867 m
.72152 .49867 L
.72152 .52242 L
.70886 .52242 L
F
.86076 .49867 m
.87342 .49867 L
.87342 .52242 L
.86076 .52242 L
F
.87342 .49867 m
.88608 .49867 L
.88608 .52242 L
.87342 .52242 L
F
.67089 .49867 m
.68354 .49867 L
.68354 .52242 L
.67089 .52242 L
F
.75949 .49867 m
.77215 .49867 L
.77215 .52242 L
.75949 .52242 L
F
.65823 .49867 m
.67089 .49867 L
.67089 .52242 L
.65823 .52242 L
F
.64557 .49867 m
.65823 .49867 L
.65823 .52242 L
.64557 .52242 L
F
.02532 .52242 m
.03797 .52242 L
.03797 .54617 L
.02532 .54617 L
F
.03797 .52242 m
.05063 .52242 L
.05063 .54617 L
.03797 .54617 L
F
.05063 .52242 m
.06329 .52242 L
.06329 .54617 L
.05063 .54617 L
F
.92405 .49867 m
.93671 .49867 L
.93671 .52242 L
.92405 .52242 L
F
.94937 .49867 m
.96203 .49867 L
.96203 .52242 L
.94937 .52242 L
F
.94937 .47493 m
.96203 .47493 L
.96203 .49867 L
.94937 .49867 L
F
.96203 .47493 m
.97468 .47493 L
.97468 .49867 L
.96203 .49867 L
F
.97468 .47493 m
.98734 .47493 L
.98734 .49867 L
.97468 .49867 L
F
0 .52242 m
.01266 .52242 L
.01266 .54617 L
0 .54617 L
F
.01266 .52242 m
.02532 .52242 L
.02532 .54617 L
.01266 .54617 L
F
.13924 .52242 m
.1519 .52242 L
.1519 .54617 L
.13924 .54617 L
F
.18987 .52242 m
.20253 .52242 L
.20253 .54617 L
.18987 .54617 L
F
.1519 .52242 m
.16456 .52242 L
.16456 .54617 L
.1519 .54617 L
F
.17722 .52242 m
.18987 .52242 L
.18987 .54617 L
.17722 .54617 L
F
.16456 .52242 m
.17722 .52242 L
.17722 .54617 L
.16456 .54617 L
F
.98734 .49867 m
1 .49867 L
1 .52242 L
.98734 .52242 L
F
.93671 .49867 m
.94937 .49867 L
.94937 .52242 L
.93671 .52242 L
F
.97468 .49867 m
.98734 .49867 L
.98734 .52242 L
.97468 .52242 L
F
.96203 .49867 m
.97468 .49867 L
.97468 .52242 L
.96203 .52242 L
F
.70886 1.0211 m
.72152 1.0211 L
.72152 1.04484 L
.70886 1.04484 L
F
.68354 1.0211 m
.6962 1.0211 L
.6962 1.04484 L
.68354 1.04484 L
F
.79747 1.0211 m
.81013 1.0211 L
.81013 1.04484 L
.79747 1.04484 L
F
.74684 1.0211 m
.75949 1.0211 L
.75949 1.04484 L
.74684 1.04484 L
F
.6962 1.0211 m
.70886 1.0211 L
.70886 1.04484 L
.6962 1.04484 L
F
.77215 1.0211 m
.78481 1.0211 L
.78481 1.04484 L
.77215 1.04484 L
F
.75949 1.0211 m
.77215 1.0211 L
.77215 1.04484 L
.75949 1.04484 L
F
.59911 1.0211 m
.60759 1.0211 L
.60759 1.04484 L
.59911 1.04484 L
F
.78481 1.0211 m
.79747 1.0211 L
.79747 1.04484 L
.78481 1.04484 L
F
.81013 1.0211 m
.82278 1.0211 L
.82278 1.04484 L
.81013 1.04484 L
F
.73418 1.16357 m
.74684 1.16357 L
.74684 1.18732 L
.73418 1.18732 L
F
.74684 1.16357 m
.75949 1.16357 L
.75949 1.18732 L
.74684 1.18732 L
F
.75949 1.16357 m
.77215 1.16357 L
.77215 1.18732 L
.75949 1.18732 L
F
.64557 1.16357 m
.65823 1.16357 L
.65823 1.18732 L
.64557 1.18732 L
F
.65823 1.16357 m
.67089 1.16357 L
.67089 1.18732 L
.65823 1.18732 L
F
.73418 1.0211 m
.74684 1.0211 L
.74684 1.04484 L
.73418 1.04484 L
F
.72152 1.0211 m
.73418 1.0211 L
.73418 1.04484 L
.72152 1.04484 L
F
.70886 1.16357 m
.72152 1.16357 L
.72152 1.18732 L
.70886 1.18732 L
F
.72152 1.16357 m
.73418 1.16357 L
.73418 1.18732 L
.72152 1.18732 L
F
.02532 1.04484 m
.03797 1.04484 L
.03797 1.06859 L
.02532 1.06859 L
F
0 1.04484 m
.01266 1.04484 L
.01266 1.06859 L
0 1.06859 L
F
.97468 1.0211 m
.98734 1.0211 L
.98734 1.04484 L
.97468 1.04484 L
F
.63291 1.0211 m
.64557 1.0211 L
.64557 1.04484 L
.63291 1.04484 L
F
.01266 1.04484 m
.02532 1.04484 L
.02532 1.06859 L
.01266 1.06859 L
F
.59911 1.04484 m
.60759 1.04484 L
.60759 1.06859 L
.59911 1.06859 L
F
.60759 1.04484 m
.62025 1.04484 L
.62025 1.06859 L
.60759 1.06859 L
F
.62025 1.04484 m
.63291 1.04484 L
.63291 1.06859 L
.62025 1.06859 L
F
.96203 1.0211 m
.97468 1.0211 L
.97468 1.04484 L
.96203 1.04484 L
F
.98734 1.0211 m
1 1.0211 L
1 1.04484 L
.98734 1.04484 L
F
.01266 1.0211 m
.02532 1.0211 L
.02532 1.04484 L
.01266 1.04484 L
F
.02532 1.0211 m
.03797 1.0211 L
.03797 1.04484 L
.02532 1.04484 L
F
.03797 1.0211 m
.05063 1.0211 L
.05063 1.04484 L
.03797 1.04484 L
F
.05063 1.0211 m
.06329 1.0211 L
.06329 1.04484 L
.05063 1.04484 L
F
.60759 1.0211 m
.62025 1.0211 L
.62025 1.04484 L
.60759 1.04484 L
F
.67089 1.0211 m
.68354 1.0211 L
.68354 1.04484 L
.67089 1.04484 L
F
.62025 1.0211 m
.63291 1.0211 L
.63291 1.04484 L
.62025 1.04484 L
F
.65823 1.0211 m
.67089 1.0211 L
.67089 1.04484 L
.65823 1.04484 L
F
.64557 1.0211 m
.65823 1.0211 L
.65823 1.04484 L
.64557 1.04484 L
F
.59888 1.16357 m
.60759 1.16357 L
.60759 1.18732 L
.59854 1.18732 L
F
.63291 1.16357 m
.64557 1.16357 L
.64557 1.18732 L
.63291 1.18732 L
F
.93671 1.13983 m
.94937 1.13983 L
.94937 1.16357 L
.93671 1.16357 L
F
.62025 1.16357 m
.63291 1.16357 L
.63291 1.18732 L
.62025 1.18732 L
F
.60759 1.16357 m
.62025 1.16357 L
.62025 1.18732 L
.60759 1.18732 L
F
.88608 1.13983 m
.89873 1.13983 L
.89873 1.16357 L
.88608 1.16357 L
F
.89873 1.13983 m
.91139 1.13983 L
.91139 1.16357 L
.89873 1.16357 L
F
.91139 1.13983 m
.92405 1.13983 L
.92405 1.16357 L
.91139 1.16357 L
F
.92405 1.13983 m
.93671 1.13983 L
.93671 1.16357 L
.92405 1.16357 L
F
.94937 1.13983 m
.96203 1.13983 L
.96203 1.16357 L
.94937 1.16357 L
F
.07595 1.16357 m
.08861 1.16357 L
.08861 1.18732 L
.07595 1.18732 L
F
.74684 1.18732 m
.75949 1.18732 L
.75949 1.21107 L
.74684 1.21107 L
F
.06329 1.16357 m
.07595 1.16357 L
.07595 1.18732 L
.06329 1.18732 L
F
.78481 1.18732 m
.79747 1.18732 L
.79747 1.21107 L
.78481 1.21107 L
F
.73418 1.18732 m
.74684 1.18732 L
.74684 1.21107 L
.73418 1.21107 L
F
.02532 1.16357 m
.03797 1.16357 L
.03797 1.18732 L
.02532 1.18732 L
F
.05063 1.16357 m
.06329 1.16357 L
.06329 1.18732 L
.05063 1.18732 L
F
.08861 1.16357 m
.10127 1.16357 L
.10127 1.18732 L
.08861 1.18732 L
F
.03797 1.16357 m
.05063 1.16357 L
.05063 1.18732 L
.03797 1.18732 L
F
.87342 1.16357 m
.88608 1.16357 L
.88608 1.18732 L
.87342 1.18732 L
F
.88608 1.16357 m
.89873 1.16357 L
.89873 1.18732 L
.88608 1.18732 L
F
.89873 1.16357 m
.91139 1.16357 L
.91139 1.18732 L
.89873 1.18732 L
F
.77215 1.16357 m
.78481 1.16357 L
.78481 1.18732 L
.77215 1.18732 L
F
.79747 1.16357 m
.81013 1.16357 L
.81013 1.18732 L
.79747 1.18732 L
F
.67089 1.16357 m
.68354 1.16357 L
.68354 1.18732 L
.67089 1.18732 L
F
.68354 1.16357 m
.6962 1.16357 L
.6962 1.18732 L
.68354 1.18732 L
F
.6962 1.16357 m
.70886 1.16357 L
.70886 1.18732 L
.6962 1.18732 L
F
.8481 1.16357 m
.86076 1.16357 L
.86076 1.18732 L
.8481 1.18732 L
F
.86076 1.16357 m
.87342 1.16357 L
.87342 1.18732 L
.86076 1.18732 L
F
.96203 1.13983 m
.97468 1.13983 L
.97468 1.16357 L
.96203 1.16357 L
F
.01266 1.16357 m
.02532 1.16357 L
.02532 1.18732 L
.01266 1.18732 L
F
.97468 1.13983 m
.98734 1.13983 L
.98734 1.16357 L
.97468 1.16357 L
F
0 1.16357 m
.01266 1.16357 L
.01266 1.18732 L
0 1.18732 L
F
.98734 1.13983 m
1 1.13983 L
1 1.16357 L
.98734 1.16357 L
F
.83544 1.16357 m
.8481 1.16357 L
.8481 1.18732 L
.83544 1.18732 L
F
.78481 1.16357 m
.79747 1.16357 L
.79747 1.18732 L
.78481 1.18732 L
F
.82278 1.16357 m
.83544 1.16357 L
.83544 1.18732 L
.82278 1.18732 L
F
.81013 1.16357 m
.82278 1.16357 L
.82278 1.18732 L
.81013 1.18732 L
F
.74684 .99735 m
.75949 .99735 L
.75949 1.0211 L
.74684 1.0211 L
F
.79747 .99735 m
.81013 .99735 L
.81013 1.0211 L
.79747 1.0211 L
F
.75949 .99735 m
.77215 .99735 L
.77215 1.0211 L
.75949 1.0211 L
F
.78481 .99735 m
.79747 .99735 L
.79747 1.0211 L
.78481 1.0211 L
F
.77215 .99735 m
.78481 .99735 L
.78481 1.0211 L
.77215 1.0211 L
F
.81013 .99735 m
.82278 .99735 L
.82278 1.0211 L
.81013 1.0211 L
F
.86076 .99735 m
.87342 .99735 L
.87342 1.0211 L
.86076 1.0211 L
F
.82278 .99735 m
.83544 .99735 L
.83544 1.0211 L
.82278 1.0211 L
F
.8481 .99735 m
.86076 .99735 L
.86076 1.0211 L
.8481 1.0211 L
F
.83544 .99735 m
.8481 .99735 L
.8481 1.0211 L
.83544 1.0211 L
F
.89873 .99735 m
.91139 .99735 L
.91139 1.0211 L
.89873 1.0211 L
F
.87342 .99735 m
.88608 .99735 L
.88608 1.0211 L
.87342 1.0211 L
F
.98734 .99735 m
1 .99735 L
1 1.0211 L
.98734 1.0211 L
F
.93671 .99735 m
.94937 .99735 L
.94937 1.0211 L
.93671 1.0211 L
F
.88608 .99735 m
.89873 .99735 L
.89873 1.0211 L
.88608 1.0211 L
F
.94937 .99735 m
.96203 .99735 L
.96203 1.0211 L
.94937 1.0211 L
F
.96203 .99735 m
.97468 .99735 L
.97468 1.0211 L
.96203 1.0211 L
F
.97468 .99735 m
.98734 .99735 L
.98734 1.0211 L
.97468 1.0211 L
F
0 1.0211 m
.01266 1.0211 L
.01266 1.04484 L
0 1.04484 L
F
.73418 1.06859 m
.74684 1.06859 L
.74684 1.09233 L
.73418 1.09233 L
F
.74684 1.06859 m
.75949 1.06859 L
.75949 1.09233 L
.74684 1.09233 L
F
.75949 1.06859 m
.77215 1.06859 L
.77215 1.09233 L
.75949 1.09233 L
F
.91139 1.06859 m
.92405 1.06859 L
.92405 1.09233 L
.91139 1.09233 L
F
.92405 1.06859 m
.93671 1.06859 L
.93671 1.09233 L
.92405 1.09233 L
F
.81013 1.06859 m
.82278 1.06859 L
.82278 1.09233 L
.81013 1.09233 L
F
.72152 1.06859 m
.73418 1.06859 L
.73418 1.09233 L
.72152 1.09233 L
F
.79747 1.06859 m
.81013 1.06859 L
.81013 1.09233 L
.79747 1.09233 L
F
.70886 1.06859 m
.72152 1.06859 L
.72152 1.09233 L
.70886 1.09233 L
F
.6962 1.06859 m
.70886 1.06859 L
.70886 1.09233 L
.6962 1.09233 L
F
.8481 1.06859 m
.86076 1.06859 L
.86076 1.09233 L
.8481 1.09233 L
F
.89873 1.06859 m
.91139 1.06859 L
.91139 1.09233 L
.89873 1.09233 L
F
.83544 1.06859 m
.8481 1.06859 L
.8481 1.09233 L
.83544 1.09233 L
F
.88608 1.06859 m
.89873 1.06859 L
.89873 1.09233 L
.88608 1.09233 L
F
.87342 1.06859 m
.88608 1.06859 L
.88608 1.09233 L
.87342 1.09233 L
F
.93671 1.06859 m
.94937 1.06859 L
.94937 1.09233 L
.93671 1.09233 L
F
.96203 1.06859 m
.97468 1.06859 L
.97468 1.09233 L
.96203 1.09233 L
F
.86076 1.06859 m
.87342 1.06859 L
.87342 1.09233 L
.86076 1.09233 L
F
.94937 1.06859 m
.96203 1.06859 L
.96203 1.09233 L
.94937 1.09233 L
F
.91139 1.0211 m
.92405 1.0211 L
.92405 1.04484 L
.91139 1.04484 L
F
.89873 1.0211 m
.91139 1.0211 L
.91139 1.04484 L
.89873 1.04484 L
F
.65823 .99735 m
.67089 .99735 L
.67089 1.0211 L
.65823 1.0211 L
F
.92405 1.0211 m
.93671 1.0211 L
.93671 1.04484 L
.92405 1.04484 L
F
.94937 1.0211 m
.96203 1.0211 L
.96203 1.04484 L
.94937 1.04484 L
F
.60759 .99735 m
.62025 .99735 L
.62025 1.0211 L
.60759 1.0211 L
F
.62025 .99735 m
.63291 .99735 L
.63291 1.0211 L
.62025 1.0211 L
F
.63291 .99735 m
.64557 .99735 L
.64557 1.0211 L
.63291 1.0211 L
F
.64557 .99735 m
.65823 .99735 L
.65823 1.0211 L
.64557 1.0211 L
F
.67089 .99735 m
.68354 .99735 L
.68354 1.0211 L
.67089 1.0211 L
F
.86076 1.0211 m
.87342 1.0211 L
.87342 1.04484 L
.86076 1.04484 L
F
.87342 1.0211 m
.88608 1.0211 L
.88608 1.04484 L
.87342 1.04484 L
F
.88608 1.0211 m
.89873 1.0211 L
.89873 1.04484 L
.88608 1.04484 L
F
.03797 1.04484 m
.05063 1.04484 L
.05063 1.06859 L
.03797 1.06859 L
F
.05063 1.04484 m
.06329 1.04484 L
.06329 1.06859 L
.05063 1.06859 L
F
.8481 1.0211 m
.86076 1.0211 L
.86076 1.04484 L
.8481 1.04484 L
F
.93671 1.0211 m
.94937 1.0211 L
.94937 1.04484 L
.93671 1.04484 L
F
.83544 1.0211 m
.8481 1.0211 L
.8481 1.04484 L
.83544 1.04484 L
F
.82278 1.0211 m
.83544 1.0211 L
.83544 1.04484 L
.82278 1.04484 L
F
.03797 .99735 m
.05063 .99735 L
.05063 1.0211 L
.03797 1.0211 L
F
.05063 .99735 m
.06329 .99735 L
.06329 1.0211 L
.05063 1.0211 L
F
.59911 .99735 m
.60759 .99735 L
.60759 1.0211 L
.59911 1.0211 L
F
.93671 .9736 m
.94937 .9736 L
.94937 .99735 L
.93671 .99735 L
F
.96203 .9736 m
.97468 .9736 L
.97468 .99735 L
.96203 .99735 L
F
.92405 .99735 m
.93671 .99735 L
.93671 1.0211 L
.92405 1.0211 L
F
.91139 .99735 m
.92405 .99735 L
.92405 1.0211 L
.91139 1.0211 L
F
.01266 .99735 m
.02532 .99735 L
.02532 1.0211 L
.01266 1.0211 L
F
.02532 .99735 m
.03797 .99735 L
.03797 1.0211 L
.02532 1.0211 L
F
.68354 .99735 m
.6962 .99735 L
.6962 1.0211 L
.68354 1.0211 L
F
.73418 .99735 m
.74684 .99735 L
.74684 1.0211 L
.73418 1.0211 L
F
.6962 .99735 m
.70886 .99735 L
.70886 1.0211 L
.6962 1.0211 L
F
.72152 .99735 m
.73418 .99735 L
.73418 1.0211 L
.72152 1.0211 L
F
.70886 .99735 m
.72152 .99735 L
.72152 1.0211 L
.70886 1.0211 L
F
0 .99735 m
.01266 .99735 L
.01266 1.0211 L
0 1.0211 L
F
.94937 .9736 m
.96203 .9736 L
.96203 .99735 L
.94937 .99735 L
F
.98734 .9736 m
1 .9736 L
1 .99735 L
.98734 .99735 L
F
.97468 .9736 m
.98734 .9736 L
.98734 .99735 L
.97468 .99735 L
F
.55696 1.686 m
.56962 1.686 L
.56962 1.70974 L
.55696 1.70974 L
F
.56962 1.686 m
.58228 1.686 L
.58228 1.70974 L
.56962 1.70974 L
F
.50633 1.686 m
.51899 1.686 L
.51899 1.70974 L
.50633 1.70974 L
F
.5443 1.686 m
.55696 1.686 L
.55696 1.70974 L
.5443 1.70974 L
F
.53165 1.686 m
.5443 1.686 L
.5443 1.70974 L
.53165 1.70974 L
F
.62025 1.686 m
.63291 1.686 L
.63291 1.70974 L
.62025 1.70974 L
F
.63291 1.686 m
.64557 1.686 L
.64557 1.70974 L
.63291 1.70974 L
F
.64557 1.686 m
.65823 1.686 L
.65823 1.70974 L
.64557 1.70974 L
F
.49367 1.686 m
.50633 1.686 L
.50633 1.70974 L
.49367 1.70974 L
F
.51899 1.686 m
.53165 1.686 L
.53165 1.70974 L
.51899 1.70974 L
F
.1519 1.686 m
.16456 1.686 L
.16456 1.70974 L
.1519 1.70974 L
F
.05063 1.686 m
.06329 1.686 L
.06329 1.70974 L
.05063 1.70974 L
F
.13924 1.686 m
.1519 1.686 L
.1519 1.70974 L
.13924 1.70974 L
F
.03797 1.686 m
.05063 1.686 L
.05063 1.70974 L
.03797 1.70974 L
F
.02532 1.686 m
.03797 1.686 L
.03797 1.70974 L
.02532 1.70974 L
F
.10127 1.686 m
.11392 1.686 L
.11392 1.70974 L
.10127 1.70974 L
F
.12658 1.686 m
.13924 1.686 L
.13924 1.70974 L
.12658 1.70974 L
F
.16456 1.686 m
.17722 1.686 L
.17722 1.70974 L
.16456 1.70974 L
F
.11392 1.686 m
.12658 1.686 L
.12658 1.70974 L
.11392 1.70974 L
F
.43038 1.686 m
.44304 1.686 L
.44304 1.70974 L
.43038 1.70974 L
F
.48101 1.686 m
.49367 1.686 L
.49367 1.70974 L
.48101 1.70974 L
F
.41772 1.686 m
.43038 1.686 L
.43038 1.70974 L
.41772 1.70974 L
F
.46835 1.686 m
.48101 1.686 L
.48101 1.70974 L
.46835 1.70974 L
F
.4557 1.686 m
.46835 1.686 L
.46835 1.70974 L
.4557 1.70974 L
F
.1519 1.73349 m
.16456 1.73349 L
.16456 1.75723 L
.1519 1.75723 L
F
.11392 1.73349 m
.12658 1.73349 L
.12658 1.75723 L
.11392 1.75723 L
F
.13924 1.73349 m
.1519 1.73349 L
.1519 1.75723 L
.13924 1.75723 L
F
.44304 1.686 m
.4557 1.686 L
.4557 1.70974 L
.44304 1.70974 L
F
.12658 1.73349 m
.13924 1.73349 L
.13924 1.75723 L
.12658 1.75723 L
F
.39241 1.686 m
.40506 1.686 L
.40506 1.70974 L
.39241 1.70974 L
F
.60759 1.686 m
.62025 1.686 L
.62025 1.70974 L
.60759 1.70974 L
F
.37975 1.686 m
.39241 1.686 L
.39241 1.70974 L
.37975 1.70974 L
F
.59494 1.686 m
.60759 1.686 L
.60759 1.70974 L
.59494 1.70974 L
F
.58228 1.686 m
.59494 1.686 L
.59494 1.70974 L
.58228 1.70974 L
F
.34177 1.686 m
.35443 1.686 L
.35443 1.70974 L
.34177 1.70974 L
F
.36709 1.686 m
.37975 1.686 L
.37975 1.70974 L
.36709 1.70974 L
F
.40506 1.686 m
.41772 1.686 L
.41772 1.70974 L
.40506 1.70974 L
F
.35443 1.686 m
.36709 1.686 L
.36709 1.70974 L
.35443 1.70974 L
F
.98734 1.686 m
1 1.686 L
1 1.70974 L
.98734 1.70974 L
F
.03797 1.70974 m
.05063 1.70974 L
.05063 1.73349 L
.03797 1.73349 L
F
.97468 1.686 m
.98734 1.686 L
.98734 1.70974 L
.97468 1.70974 L
F
.02532 1.70974 m
.03797 1.70974 L
.03797 1.73349 L
.02532 1.73349 L
F
.01266 1.70974 m
.02532 1.70974 L
.02532 1.73349 L
.01266 1.73349 L
F
.11392 1.70974 m
.12658 1.70974 L
.12658 1.73349 L
.11392 1.73349 L
F
.08861 1.70974 m
.10127 1.70974 L
.10127 1.73349 L
.08861 1.73349 L
F
.06329 1.70974 m
.07595 1.70974 L
.07595 1.73349 L
.06329 1.73349 L
F
0 1.70974 m
.01266 1.70974 L
.01266 1.73349 L
0 1.73349 L
F
.10127 1.70974 m
.11392 1.70974 L
.11392 1.73349 L
.10127 1.73349 L
F
.26582 1.70974 m
.27848 1.70974 L
.27848 1.73349 L
.26582 1.73349 L
F
.1519 1.70974 m
.16456 1.70974 L
.16456 1.73349 L
.1519 1.73349 L
F
.25316 1.70974 m
.26582 1.70974 L
.26582 1.73349 L
.25316 1.73349 L
F
.13924 1.70974 m
.1519 1.70974 L
.1519 1.73349 L
.13924 1.73349 L
F
.12658 1.70974 m
.13924 1.70974 L
.13924 1.73349 L
.12658 1.73349 L
F
.21519 1.70974 m
.22785 1.70974 L
.22785 1.73349 L
.21519 1.73349 L
F
.24051 1.70974 m
.25316 1.70974 L
.25316 1.73349 L
.24051 1.73349 L
F
.27848 1.70974 m
.29114 1.70974 L
.29114 1.73349 L
.27848 1.73349 L
F
.22785 1.70974 m
.24051 1.70974 L
.24051 1.73349 L
.22785 1.73349 L
F
.27848 1.686 m
.29114 1.686 L
.29114 1.70974 L
.27848 1.70974 L
F
.32911 1.686 m
.34177 1.686 L
.34177 1.70974 L
.32911 1.70974 L
F
.26582 1.686 m
.27848 1.686 L
.27848 1.70974 L
.26582 1.70974 L
F
.31646 1.686 m
.32911 1.686 L
.32911 1.70974 L
.31646 1.70974 L
F
.3038 1.686 m
.31646 1.686 L
.31646 1.70974 L
.3038 1.70974 L
F
.06329 1.686 m
.07595 1.686 L
.07595 1.70974 L
.06329 1.70974 L
F
.08861 1.686 m
.10127 1.686 L
.10127 1.70974 L
.08861 1.70974 L
F
.29114 1.686 m
.3038 1.686 L
.3038 1.70974 L
.29114 1.70974 L
F
.07595 1.686 m
.08861 1.686 L
.08861 1.70974 L
.07595 1.70974 L
F
.21519 1.686 m
.22785 1.686 L
.22785 1.70974 L
.21519 1.70974 L
F
.22785 1.686 m
.24051 1.686 L
.24051 1.70974 L
.22785 1.70974 L
F
.25316 1.686 m
.26582 1.686 L
.26582 1.70974 L
.25316 1.70974 L
F
.05063 1.70974 m
.06329 1.70974 L
.06329 1.73349 L
.05063 1.73349 L
F
.07595 1.70974 m
.08861 1.70974 L
.08861 1.73349 L
.07595 1.73349 L
F
.17722 1.686 m
.18987 1.686 L
.18987 1.70974 L
.17722 1.70974 L
F
.20253 1.686 m
.21519 1.686 L
.21519 1.70974 L
.20253 1.70974 L
F
.24051 1.686 m
.25316 1.686 L
.25316 1.70974 L
.24051 1.70974 L
F
.18987 1.686 m
.20253 1.686 L
.20253 1.70974 L
.18987 1.70974 L
F
.50633 1.70974 m
.51899 1.70974 L
.51899 1.73349 L
.50633 1.73349 L
F
.51899 1.70974 m
.53165 1.70974 L
.53165 1.73349 L
.51899 1.73349 L
F
.4557 1.70974 m
.46835 1.70974 L
.46835 1.73349 L
.4557 1.73349 L
F
.49367 1.70974 m
.50633 1.70974 L
.50633 1.73349 L
.49367 1.73349 L
F
.48101 1.70974 m
.49367 1.70974 L
.49367 1.73349 L
.48101 1.73349 L
F
.56962 1.70974 m
.58228 1.70974 L
.58228 1.73349 L
.56962 1.73349 L
F
.58228 1.70974 m
.59494 1.70974 L
.59494 1.73349 L
.58228 1.73349 L
F
.59494 1.70974 m
.60759 1.70974 L
.60759 1.73349 L
.59494 1.73349 L
F
.44304 1.70974 m
.4557 1.70974 L
.4557 1.73349 L
.44304 1.73349 L
F
.46835 1.70974 m
.48101 1.70974 L
.48101 1.73349 L
.46835 1.73349 L
F
.36709 1.73349 m
.37975 1.73349 L
.37975 1.75723 L
.36709 1.75723 L
F
.26582 1.73349 m
.27848 1.73349 L
.27848 1.75723 L
.26582 1.75723 L
F
.35443 1.73349 m
.36709 1.73349 L
.36709 1.75723 L
.35443 1.75723 L
F
.25316 1.73349 m
.26582 1.73349 L
.26582 1.75723 L
.25316 1.75723 L
F
.24051 1.73349 m
.25316 1.73349 L
.25316 1.75723 L
.24051 1.75723 L
F
.31646 1.73349 m
.32911 1.73349 L
.32911 1.75723 L
.31646 1.75723 L
F
.34177 1.73349 m
.35443 1.73349 L
.35443 1.75723 L
.34177 1.75723 L
F
.37975 1.73349 m
.39241 1.73349 L
.39241 1.75723 L
.37975 1.75723 L
F
.32911 1.73349 m
.34177 1.73349 L
.34177 1.75723 L
.32911 1.75723 L
F
.37975 1.70974 m
.39241 1.70974 L
.39241 1.73349 L
.37975 1.73349 L
F
.43038 1.70974 m
.44304 1.70974 L
.44304 1.73349 L
.43038 1.73349 L
F
.36709 1.70974 m
.37975 1.70974 L
.37975 1.73349 L
.36709 1.73349 L
F
.41772 1.70974 m
.43038 1.70974 L
.43038 1.73349 L
.41772 1.73349 L
F
.40506 1.70974 m
.41772 1.70974 L
.41772 1.73349 L
.40506 1.73349 L
F
.79747 1.70974 m
.81013 1.70974 L
.81013 1.73349 L
.79747 1.73349 L
F
.83544 1.70974 m
.8481 1.70974 L
.8481 1.73349 L
.83544 1.73349 L
F
.39241 1.70974 m
.40506 1.70974 L
.40506 1.73349 L
.39241 1.73349 L
F
.81013 1.70974 m
.82278 1.70974 L
.82278 1.73349 L
.81013 1.73349 L
F
.34177 1.70974 m
.35443 1.70974 L
.35443 1.73349 L
.34177 1.73349 L
F
.55696 1.70974 m
.56962 1.70974 L
.56962 1.73349 L
.55696 1.73349 L
F
.32911 1.70974 m
.34177 1.70974 L
.34177 1.73349 L
.32911 1.73349 L
F
.5443 1.70974 m
.55696 1.70974 L
.55696 1.73349 L
.5443 1.73349 L
F
.53165 1.70974 m
.5443 1.70974 L
.5443 1.73349 L
.53165 1.73349 L
F
.29114 1.70974 m
.3038 1.70974 L
.3038 1.73349 L
.29114 1.73349 L
F
.31646 1.70974 m
.32911 1.70974 L
.32911 1.73349 L
.31646 1.73349 L
F
.35443 1.70974 m
.36709 1.70974 L
.36709 1.73349 L
.35443 1.73349 L
F
.3038 1.70974 m
.31646 1.70974 L
.31646 1.73349 L
.3038 1.73349 L
F
.93671 1.70974 m
.94937 1.70974 L
.94937 1.73349 L
.93671 1.73349 L
F
.98734 1.70974 m
1 1.70974 L
1 1.73349 L
.98734 1.73349 L
F
.92405 1.70974 m
.93671 1.70974 L
.93671 1.73349 L
.92405 1.73349 L
F
.97468 1.70974 m
.98734 1.70974 L
.98734 1.73349 L
.97468 1.73349 L
F
.96203 1.70974 m
.97468 1.70974 L
.97468 1.73349 L
.96203 1.73349 L
F
.06329 1.73349 m
.07595 1.73349 L
.07595 1.75723 L
.06329 1.75723 L
F
.03797 1.73349 m
.05063 1.73349 L
.05063 1.75723 L
.03797 1.75723 L
F
.01266 1.73349 m
.02532 1.73349 L
.02532 1.75723 L
.01266 1.75723 L
F
.94937 1.70974 m
.96203 1.70974 L
.96203 1.73349 L
.94937 1.73349 L
F
.05063 1.73349 m
.06329 1.73349 L
.06329 1.75723 L
.05063 1.75723 L
F
.21519 1.73349 m
.22785 1.73349 L
.22785 1.75723 L
.21519 1.75723 L
F
.10127 1.73349 m
.11392 1.73349 L
.11392 1.75723 L
.10127 1.75723 L
F
.20253 1.73349 m
.21519 1.73349 L
.21519 1.75723 L
.20253 1.75723 L
F
.08861 1.73349 m
.10127 1.73349 L
.10127 1.75723 L
.08861 1.75723 L
F
.07595 1.73349 m
.08861 1.73349 L
.08861 1.75723 L
.07595 1.75723 L
F
.16456 1.73349 m
.17722 1.73349 L
.17722 1.75723 L
.16456 1.75723 L
F
.18987 1.73349 m
.20253 1.73349 L
.20253 1.75723 L
.18987 1.75723 L
F
.22785 1.73349 m
.24051 1.73349 L
.24051 1.75723 L
.22785 1.75723 L
F
.17722 1.73349 m
.18987 1.73349 L
.18987 1.75723 L
.17722 1.75723 L
F
.49367 1.73349 m
.50633 1.73349 L
.50633 1.75723 L
.49367 1.75723 L
F
.5443 1.73349 m
.55696 1.73349 L
.55696 1.75723 L
.5443 1.75723 L
F
.48101 1.73349 m
.49367 1.73349 L
.49367 1.75723 L
.48101 1.75723 L
F
.53165 1.73349 m
.5443 1.73349 L
.5443 1.75723 L
.53165 1.75723 L
F
.51899 1.73349 m
.53165 1.73349 L
.53165 1.75723 L
.51899 1.75723 L
F
.27848 1.73349 m
.29114 1.73349 L
.29114 1.75723 L
.27848 1.75723 L
F
.3038 1.73349 m
.31646 1.73349 L
.31646 1.75723 L
.3038 1.75723 L
F
.50633 1.73349 m
.51899 1.73349 L
.51899 1.75723 L
.50633 1.75723 L
F
.29114 1.73349 m
.3038 1.73349 L
.3038 1.75723 L
.29114 1.75723 L
F
.43038 1.73349 m
.44304 1.73349 L
.44304 1.75723 L
.43038 1.75723 L
F
.44304 1.73349 m
.4557 1.73349 L
.4557 1.75723 L
.44304 1.75723 L
F
.46835 1.73349 m
.48101 1.73349 L
.48101 1.75723 L
.46835 1.75723 L
F
0 1.73349 m
.01266 1.73349 L
.01266 1.75723 L
0 1.75723 L
F
.02532 1.73349 m
.03797 1.73349 L
.03797 1.75723 L
.02532 1.75723 L
F
.39241 1.73349 m
.40506 1.73349 L
.40506 1.75723 L
.39241 1.75723 L
F
.41772 1.73349 m
.43038 1.73349 L
.43038 1.75723 L
.41772 1.75723 L
F
.4557 1.73349 m
.46835 1.73349 L
.46835 1.75723 L
.4557 1.75723 L
F
.40506 1.73349 m
.41772 1.73349 L
.41772 1.75723 L
.40506 1.75723 L
F
.93671 1.78098 m
.94937 1.78098 L
.94937 1.80473 L
.93671 1.80473 L
F
.98734 1.78098 m
1 1.78098 L
1 1.80473 L
.98734 1.80473 L
F
.92405 1.78098 m
.93671 1.78098 L
.93671 1.80473 L
.92405 1.80473 L
F
.96203 1.78098 m
.97468 1.78098 L
.97468 1.80473 L
.96203 1.80473 L
F
0 1.80473 m
.01266 1.80473 L
.01266 1.82847 L
0 1.82847 L
F
.07595 1.80473 m
.08861 1.80473 L
.08861 1.82847 L
.07595 1.82847 L
F
.05063 1.80473 m
.06329 1.80473 L
.06329 1.82847 L
.05063 1.82847 L
F
.02532 1.80473 m
.03797 1.80473 L
.03797 1.82847 L
.02532 1.82847 L
F
.94937 1.78098 m
.96203 1.78098 L
.96203 1.80473 L
.94937 1.80473 L
F
.06329 1.80473 m
.07595 1.80473 L
.07595 1.82847 L
.06329 1.82847 L
F
.56962 1.78098 m
.58228 1.78098 L
.58228 1.80473 L
.56962 1.80473 L
F
.58228 1.78098 m
.59494 1.78098 L
.59494 1.80473 L
.58228 1.80473 L
F
.59494 1.78098 m
.60759 1.78098 L
.60759 1.80473 L
.59494 1.80473 L
F
.4557 1.78098 m
.46835 1.78098 L
.46835 1.80473 L
.4557 1.80473 L
F
.48101 1.78098 m
.49367 1.78098 L
.49367 1.80473 L
.48101 1.80473 L
F
.55696 1.78098 m
.56962 1.78098 L
.56962 1.80473 L
.55696 1.80473 L
F
.97468 1.78098 m
.98734 1.78098 L
.98734 1.80473 L
.97468 1.80473 L
F
.5443 1.78098 m
.55696 1.78098 L
.55696 1.80473 L
.5443 1.80473 L
F
.53165 1.78098 m
.5443 1.78098 L
.5443 1.80473 L
.53165 1.80473 L
F
.8481 1.78098 m
.86076 1.78098 L
.86076 1.80473 L
.8481 1.80473 L
F
.86076 1.78098 m
.87342 1.78098 L
.87342 1.80473 L
.86076 1.80473 L
F
.87342 1.78098 m
.88608 1.78098 L
.88608 1.80473 L
.87342 1.80473 L
F
.88608 1.78098 m
.89873 1.78098 L
.89873 1.80473 L
.88608 1.80473 L
F
.91139 1.78098 m
.92405 1.78098 L
.92405 1.80473 L
.91139 1.80473 L
F
.03797 1.78098 m
.05063 1.78098 L
.05063 1.80473 L
.03797 1.80473 L
F
.05063 1.78098 m
.06329 1.78098 L
.06329 1.80473 L
.05063 1.80473 L
F
.98734 1.75723 m
1 1.75723 L
1 1.78098 L
.98734 1.78098 L
F
.02532 1.78098 m
.03797 1.78098 L
.03797 1.80473 L
.02532 1.80473 L
F
.01266 1.78098 m
.02532 1.78098 L
.02532 1.80473 L
.01266 1.80473 L
F
.81013 1.78098 m
.82278 1.78098 L
.82278 1.80473 L
.81013 1.80473 L
F
.82278 1.78098 m
.83544 1.78098 L
.83544 1.80473 L
.82278 1.80473 L
F
.83544 1.78098 m
.8481 1.78098 L
.8481 1.80473 L
.83544 1.80473 L
F
.01266 1.80473 m
.02532 1.80473 L
.02532 1.82847 L
.01266 1.82847 L
F
.03797 1.80473 m
.05063 1.80473 L
.05063 1.82847 L
.03797 1.82847 L
F
.79747 1.78098 m
.81013 1.78098 L
.81013 1.80473 L
.79747 1.80473 L
F
.89873 1.78098 m
.91139 1.78098 L
.91139 1.80473 L
.89873 1.80473 L
F
.78481 1.78098 m
.79747 1.78098 L
.79747 1.80473 L
.78481 1.80473 L
F
.77215 1.78098 m
.78481 1.78098 L
.78481 1.80473 L
.77215 1.80473 L
F
.87342 1.73349 m
.88608 1.73349 L
.88608 1.75723 L
.87342 1.75723 L
F
.88608 1.73349 m
.89873 1.73349 L
.89873 1.75723 L
.88608 1.75723 L
F
.89873 1.73349 m
.91139 1.73349 L
.91139 1.75723 L
.89873 1.75723 L
F
.91139 1.73349 m
.92405 1.73349 L
.92405 1.75723 L
.91139 1.75723 L
F
.93671 1.73349 m
.94937 1.73349 L
.94937 1.75723 L
.93671 1.75723 L
F
.96203 1.73349 m
.97468 1.73349 L
.97468 1.75723 L
.96203 1.75723 L
F
.01266 1.75723 m
.02532 1.75723 L
.02532 1.78098 L
.01266 1.78098 L
F
.94937 1.73349 m
.96203 1.73349 L
.96203 1.75723 L
.94937 1.75723 L
F
0 1.75723 m
.01266 1.75723 L
.01266 1.78098 L
0 1.78098 L
F
.98734 1.73349 m
1 1.73349 L
1 1.75723 L
.98734 1.75723 L
F
.1519 1.75723 m
.16456 1.75723 L
.16456 1.78098 L
.1519 1.78098 L
F
.16456 1.75723 m
.17722 1.75723 L
.17722 1.78098 L
.16456 1.78098 L
F
.17722 1.75723 m
.18987 1.75723 L
.18987 1.78098 L
.17722 1.78098 L
F
.02532 1.75723 m
.03797 1.75723 L
.03797 1.78098 L
.02532 1.78098 L
F
.05063 1.75723 m
.06329 1.75723 L
.06329 1.78098 L
.05063 1.78098 L
F
.13924 1.75723 m
.1519 1.75723 L
.1519 1.78098 L
.13924 1.78098 L
F
.92405 1.73349 m
.93671 1.73349 L
.93671 1.75723 L
.92405 1.75723 L
F
.12658 1.75723 m
.13924 1.75723 L
.13924 1.78098 L
.12658 1.78098 L
F
.11392 1.75723 m
.12658 1.75723 L
.12658 1.78098 L
.11392 1.78098 L
F
.6962 1.78098 m
.70886 1.78098 L
.70886 1.80473 L
.6962 1.80473 L
F
.70886 1.78098 m
.72152 1.78098 L
.72152 1.80473 L
.70886 1.80473 L
F
.72152 1.78098 m
.73418 1.78098 L
.73418 1.80473 L
.72152 1.80473 L
F
.73418 1.78098 m
.74684 1.78098 L
.74684 1.80473 L
.73418 1.80473 L
F
.75949 1.78098 m
.77215 1.78098 L
.77215 1.80473 L
.75949 1.80473 L
F
.51899 1.78098 m
.53165 1.78098 L
.53165 1.80473 L
.51899 1.80473 L
F
.46835 1.78098 m
.48101 1.78098 L
.48101 1.80473 L
.46835 1.80473 L
F
.50633 1.78098 m
.51899 1.78098 L
.51899 1.80473 L
.50633 1.80473 L
F
.49367 1.78098 m
.50633 1.78098 L
.50633 1.80473 L
.49367 1.80473 L
F
.68354 1.78098 m
.6962 1.78098 L
.6962 1.80473 L
.68354 1.80473 L
F
.64557 1.78098 m
.65823 1.78098 L
.65823 1.80473 L
.64557 1.80473 L
F
.67089 1.78098 m
.68354 1.78098 L
.68354 1.80473 L
.67089 1.80473 L
F
.97468 1.73349 m
.98734 1.73349 L
.98734 1.75723 L
.97468 1.75723 L
F
.65823 1.78098 m
.67089 1.78098 L
.67089 1.80473 L
.65823 1.80473 L
F
.63291 1.78098 m
.64557 1.78098 L
.64557 1.80473 L
.63291 1.80473 L
F
.74684 1.78098 m
.75949 1.78098 L
.75949 1.80473 L
.74684 1.80473 L
F
.62025 1.78098 m
.63291 1.78098 L
.63291 1.80473 L
.62025 1.80473 L
F
.60759 1.78098 m
.62025 1.78098 L
.62025 1.80473 L
.60759 1.80473 L
F
.82278 1.686 m
.83544 1.686 L
.83544 1.70974 L
.82278 1.70974 L
F
.87342 1.686 m
.88608 1.686 L
.88608 1.70974 L
.87342 1.70974 L
F
.81013 1.686 m
.82278 1.686 L
.82278 1.70974 L
.81013 1.70974 L
F
.8481 1.686 m
.86076 1.686 L
.86076 1.70974 L
.8481 1.70974 L
F
.88608 1.686 m
.89873 1.686 L
.89873 1.70974 L
.88608 1.70974 L
F
.96203 1.686 m
.97468 1.686 L
.97468 1.70974 L
.96203 1.70974 L
F
.93671 1.686 m
.94937 1.686 L
.94937 1.70974 L
.93671 1.70974 L
F
.91139 1.686 m
.92405 1.686 L
.92405 1.70974 L
.91139 1.70974 L
F
.83544 1.686 m
.8481 1.686 L
.8481 1.70974 L
.83544 1.70974 L
F
.94937 1.686 m
.96203 1.686 L
.96203 1.70974 L
.94937 1.70974 L
F
.25316 1.78098 m
.26582 1.78098 L
.26582 1.80473 L
.25316 1.80473 L
F
.26582 1.78098 m
.27848 1.78098 L
.27848 1.80473 L
.26582 1.80473 L
F
.27848 1.78098 m
.29114 1.78098 L
.29114 1.80473 L
.27848 1.80473 L
F
.13924 1.78098 m
.1519 1.78098 L
.1519 1.80473 L
.13924 1.80473 L
F
.16456 1.78098 m
.17722 1.78098 L
.17722 1.80473 L
.16456 1.80473 L
F
.24051 1.78098 m
.25316 1.78098 L
.25316 1.80473 L
.24051 1.80473 L
F
.86076 1.686 m
.87342 1.686 L
.87342 1.70974 L
.86076 1.70974 L
F
.22785 1.78098 m
.24051 1.78098 L
.24051 1.80473 L
.22785 1.80473 L
F
.21519 1.78098 m
.22785 1.78098 L
.22785 1.80473 L
.21519 1.80473 L
F
.73418 1.686 m
.74684 1.686 L
.74684 1.70974 L
.73418 1.70974 L
F
.74684 1.686 m
.75949 1.686 L
.75949 1.70974 L
.74684 1.70974 L
F
.75949 1.686 m
.77215 1.686 L
.77215 1.70974 L
.75949 1.70974 L
F
.77215 1.686 m
.78481 1.686 L
.78481 1.70974 L
.77215 1.70974 L
F
.79747 1.686 m
.81013 1.686 L
.81013 1.70974 L
.79747 1.70974 L
F
.20253 1.70974 m
.21519 1.70974 L
.21519 1.73349 L
.20253 1.73349 L
F
.18987 1.70974 m
.20253 1.70974 L
.20253 1.73349 L
.18987 1.73349 L
F
.17722 1.70974 m
.18987 1.70974 L
.18987 1.73349 L
.17722 1.73349 L
F
.16456 1.70974 m
.17722 1.70974 L
.17722 1.73349 L
.16456 1.73349 L
F
.6962 1.686 m
.70886 1.686 L
.70886 1.70974 L
.6962 1.70974 L
F
.70886 1.686 m
.72152 1.686 L
.72152 1.70974 L
.70886 1.70974 L
F
.72152 1.686 m
.73418 1.686 L
.73418 1.70974 L
.72152 1.70974 L
F
.89873 1.686 m
.91139 1.686 L
.91139 1.70974 L
.89873 1.70974 L
F
.92405 1.686 m
.93671 1.686 L
.93671 1.70974 L
.92405 1.70974 L
F
.68354 1.686 m
.6962 1.686 L
.6962 1.70974 L
.68354 1.70974 L
F
.78481 1.686 m
.79747 1.686 L
.79747 1.70974 L
.78481 1.70974 L
F
.67089 1.686 m
.68354 1.686 L
.68354 1.70974 L
.67089 1.70974 L
F
.65823 1.686 m
.67089 1.686 L
.67089 1.70974 L
.65823 1.70974 L
F
.82278 1.75723 m
.83544 1.75723 L
.83544 1.78098 L
.82278 1.78098 L
F
.83544 1.75723 m
.8481 1.75723 L
.8481 1.78098 L
.83544 1.78098 L
F
.8481 1.75723 m
.86076 1.75723 L
.86076 1.78098 L
.8481 1.78098 L
F
.86076 1.75723 m
.87342 1.75723 L
.87342 1.78098 L
.86076 1.78098 L
F
.88608 1.75723 m
.89873 1.75723 L
.89873 1.78098 L
.88608 1.78098 L
F
.91139 1.75723 m
.92405 1.75723 L
.92405 1.78098 L
.91139 1.78098 L
F
.96203 1.75723 m
.97468 1.75723 L
.97468 1.78098 L
.96203 1.78098 L
F
.89873 1.75723 m
.91139 1.75723 L
.91139 1.78098 L
.89873 1.78098 L
F
.94937 1.75723 m
.96203 1.75723 L
.96203 1.78098 L
.94937 1.78098 L
F
.93671 1.75723 m
.94937 1.75723 L
.94937 1.78098 L
.93671 1.78098 L
F
.10127 1.78098 m
.11392 1.78098 L
.11392 1.80473 L
.10127 1.80473 L
F
.11392 1.78098 m
.12658 1.78098 L
.12658 1.80473 L
.11392 1.80473 L
F
.12658 1.78098 m
.13924 1.78098 L
.13924 1.80473 L
.12658 1.80473 L
F
.97468 1.75723 m
.98734 1.75723 L
.98734 1.78098 L
.97468 1.78098 L
F
0 1.78098 m
.01266 1.78098 L
.01266 1.80473 L
0 1.80473 L
F
.08861 1.78098 m
.10127 1.78098 L
.10127 1.80473 L
.08861 1.80473 L
F
.87342 1.75723 m
.88608 1.75723 L
.88608 1.78098 L
.87342 1.78098 L
F
.07595 1.78098 m
.08861 1.78098 L
.08861 1.80473 L
.07595 1.80473 L
F
.06329 1.78098 m
.07595 1.78098 L
.07595 1.80473 L
.06329 1.80473 L
F
.37975 1.78098 m
.39241 1.78098 L
.39241 1.80473 L
.37975 1.80473 L
F
.39241 1.78098 m
.40506 1.78098 L
.40506 1.80473 L
.39241 1.80473 L
F
.40506 1.78098 m
.41772 1.78098 L
.41772 1.80473 L
.40506 1.80473 L
F
.41772 1.78098 m
.43038 1.78098 L
.43038 1.80473 L
.41772 1.80473 L
F
.44304 1.78098 m
.4557 1.78098 L
.4557 1.80473 L
.44304 1.80473 L
F
.20253 1.78098 m
.21519 1.78098 L
.21519 1.80473 L
.20253 1.80473 L
F
.1519 1.78098 m
.16456 1.78098 L
.16456 1.80473 L
.1519 1.80473 L
F
.18987 1.78098 m
.20253 1.78098 L
.20253 1.80473 L
.18987 1.80473 L
F
.17722 1.78098 m
.18987 1.78098 L
.18987 1.80473 L
.17722 1.80473 L
F
.36709 1.78098 m
.37975 1.78098 L
.37975 1.80473 L
.36709 1.80473 L
F
.32911 1.78098 m
.34177 1.78098 L
.34177 1.80473 L
.32911 1.80473 L
F
.35443 1.78098 m
.36709 1.78098 L
.36709 1.80473 L
.35443 1.80473 L
F
.92405 1.75723 m
.93671 1.75723 L
.93671 1.78098 L
.92405 1.78098 L
F
.34177 1.78098 m
.35443 1.78098 L
.35443 1.80473 L
.34177 1.80473 L
F
.31646 1.78098 m
.32911 1.78098 L
.32911 1.80473 L
.31646 1.80473 L
F
.43038 1.78098 m
.44304 1.78098 L
.44304 1.80473 L
.43038 1.80473 L
F
.3038 1.78098 m
.31646 1.78098 L
.31646 1.80473 L
.3038 1.80473 L
F
.29114 1.78098 m
.3038 1.78098 L
.3038 1.80473 L
.29114 1.80473 L
F
.35443 1.80473 m
.36709 1.80473 L
.36709 1.82847 L
.35443 1.82847 L
F
.32911 1.80473 m
.34177 1.80473 L
.34177 1.82847 L
.32911 1.82847 L
F
.13924 1.80473 m
.1519 1.80473 L
.1519 1.82847 L
.13924 1.82847 L
F
.39241 1.80473 m
.40506 1.80473 L
.40506 1.82847 L
.39241 1.82847 L
F
.34177 1.80473 m
.35443 1.80473 L
.35443 1.82847 L
.34177 1.82847 L
F
.08861 1.80473 m
.10127 1.80473 L
.10127 1.82847 L
.08861 1.82847 L
F
.10127 1.80473 m
.11392 1.80473 L
.11392 1.82847 L
.10127 1.82847 L
F
.11392 1.80473 m
.12658 1.80473 L
.12658 1.82847 L
.11392 1.82847 L
F
.12658 1.80473 m
.13924 1.80473 L
.13924 1.82847 L
.12658 1.82847 L
F
.1519 1.80473 m
.16456 1.80473 L
.16456 1.82847 L
.1519 1.82847 L
F
.3038 1.80473 m
.31646 1.80473 L
.31646 1.82847 L
.3038 1.82847 L
F
.31646 1.80473 m
.32911 1.80473 L
.32911 1.82847 L
.31646 1.82847 L
F
.25316 1.80473 m
.26582 1.80473 L
.26582 1.82847 L
.25316 1.82847 L
F
.29114 1.80473 m
.3038 1.80473 L
.3038 1.82847 L
.29114 1.82847 L
F
.27848 1.80473 m
.29114 1.80473 L
.29114 1.82847 L
.27848 1.82847 L
F
.37975 1.80473 m
.39241 1.80473 L
.39241 1.82847 L
.37975 1.82847 L
F
.36709 1.80473 m
.37975 1.80473 L
.37975 1.82847 L
.36709 1.82847 L
F
.24051 1.80473 m
.25316 1.80473 L
.25316 1.82847 L
.24051 1.82847 L
F
.26582 1.80473 m
.27848 1.80473 L
.27848 1.82847 L
.26582 1.82847 L
F
.58228 1.80473 m
.59494 1.80473 L
.59494 1.82847 L
.58228 1.82847 L
F
.55696 1.80473 m
.56962 1.80473 L
.56962 1.82847 L
.55696 1.82847 L
F
.6962 1.80473 m
.70886 1.80473 L
.70886 1.82847 L
.6962 1.82847 L
F
.62025 1.80473 m
.63291 1.80473 L
.63291 1.82847 L
.62025 1.82847 L
F
.56962 1.80473 m
.58228 1.80473 L
.58228 1.82847 L
.56962 1.82847 L
F
.64557 1.80473 m
.65823 1.80473 L
.65823 1.82847 L
.64557 1.82847 L
F
.65823 1.80473 m
.67089 1.80473 L
.67089 1.82847 L
.65823 1.82847 L
F
.67089 1.80473 m
.68354 1.80473 L
.68354 1.82847 L
.67089 1.82847 L
F
.68354 1.80473 m
.6962 1.80473 L
.6962 1.82847 L
.68354 1.82847 L
F
.70886 1.80473 m
.72152 1.80473 L
.72152 1.82847 L
.70886 1.82847 L
F
.17722 1.80473 m
.18987 1.80473 L
.18987 1.82847 L
.17722 1.82847 L
F
.22785 1.80473 m
.24051 1.80473 L
.24051 1.82847 L
.22785 1.82847 L
F
.16456 1.80473 m
.17722 1.80473 L
.17722 1.82847 L
.16456 1.82847 L
F
.21519 1.80473 m
.22785 1.80473 L
.22785 1.82847 L
.21519 1.82847 L
F
.20253 1.80473 m
.21519 1.80473 L
.21519 1.82847 L
.20253 1.82847 L
F
.59494 1.80473 m
.60759 1.80473 L
.60759 1.82847 L
.59494 1.82847 L
F
.63291 1.80473 m
.64557 1.80473 L
.64557 1.82847 L
.63291 1.82847 L
F
.18987 1.80473 m
.20253 1.80473 L
.20253 1.82847 L
.18987 1.82847 L
F
.60759 1.80473 m
.62025 1.80473 L
.62025 1.82847 L
.60759 1.82847 L
F
.24051 1.82847 m
.25316 1.82847 L
.25316 1.85222 L
.24051 1.85222 L
F
.82278 1.80473 m
.83544 1.80473 L
.83544 1.82847 L
.82278 1.82847 L
F
.22785 1.82847 m
.24051 1.82847 L
.24051 1.85222 L
.22785 1.85222 L
F
.81013 1.80473 m
.82278 1.80473 L
.82278 1.82847 L
.81013 1.82847 L
F
.79747 1.80473 m
.81013 1.80473 L
.81013 1.82847 L
.79747 1.82847 L
F
.18987 1.82847 m
.20253 1.82847 L
.20253 1.85222 L
.18987 1.85222 L
F
.20253 1.82847 m
.21519 1.82847 L
.21519 1.85222 L
.20253 1.85222 L
F
.21519 1.82847 m
.22785 1.82847 L
.22785 1.85222 L
.21519 1.85222 L
F
.26582 1.82847 m
.27848 1.82847 L
.27848 1.85222 L
.26582 1.85222 L
F
.25316 1.82847 m
.26582 1.82847 L
.26582 1.85222 L
.25316 1.85222 L
F
.73418 1.80473 m
.74684 1.80473 L
.74684 1.82847 L
.73418 1.82847 L
F
.78481 1.80473 m
.79747 1.80473 L
.79747 1.82847 L
.78481 1.82847 L
F
.72152 1.80473 m
.73418 1.80473 L
.73418 1.82847 L
.72152 1.82847 L
F
.77215 1.80473 m
.78481 1.80473 L
.78481 1.82847 L
.77215 1.82847 L
F
.75949 1.80473 m
.77215 1.80473 L
.77215 1.82847 L
.75949 1.82847 L
F
.83544 1.80473 m
.8481 1.80473 L
.8481 1.82847 L
.83544 1.82847 L
F
.86076 1.80473 m
.87342 1.80473 L
.87342 1.82847 L
.86076 1.82847 L
F
.74684 1.80473 m
.75949 1.80473 L
.75949 1.82847 L
.74684 1.82847 L
F
.8481 1.80473 m
.86076 1.80473 L
.86076 1.82847 L
.8481 1.82847 L
F
.16456 1.82847 m
.17722 1.82847 L
.17722 1.85222 L
.16456 1.85222 L
F
.06329 1.82847 m
.07595 1.82847 L
.07595 1.85222 L
.06329 1.85222 L
F
.1519 1.82847 m
.16456 1.82847 L
.16456 1.85222 L
.1519 1.85222 L
F
.05063 1.82847 m
.06329 1.82847 L
.06329 1.85222 L
.05063 1.85222 L
F
.03797 1.82847 m
.05063 1.82847 L
.05063 1.85222 L
.03797 1.85222 L
F
.11392 1.82847 m
.12658 1.82847 L
.12658 1.85222 L
.11392 1.85222 L
F
.13924 1.82847 m
.1519 1.82847 L
.1519 1.85222 L
.13924 1.85222 L
F
.17722 1.82847 m
.18987 1.82847 L
.18987 1.85222 L
.17722 1.85222 L
F
.12658 1.82847 m
.13924 1.82847 L
.13924 1.85222 L
.12658 1.85222 L
F
.29114 1.82847 m
.3038 1.82847 L
.3038 1.85222 L
.29114 1.85222 L
F
.34177 1.82847 m
.35443 1.82847 L
.35443 1.85222 L
.34177 1.85222 L
F
.27848 1.82847 m
.29114 1.82847 L
.29114 1.85222 L
.27848 1.85222 L
F
.32911 1.82847 m
.34177 1.82847 L
.34177 1.85222 L
.32911 1.85222 L
F
.31646 1.82847 m
.32911 1.82847 L
.32911 1.85222 L
.31646 1.85222 L
F
.07595 1.82847 m
.08861 1.82847 L
.08861 1.85222 L
.07595 1.85222 L
F
.10127 1.82847 m
.11392 1.82847 L
.11392 1.85222 L
.10127 1.85222 L
F
.3038 1.82847 m
.31646 1.82847 L
.31646 1.85222 L
.3038 1.85222 L
F
.08861 1.82847 m
.10127 1.82847 L
.10127 1.85222 L
.08861 1.85222 L
F
.55696 1.85222 m
.56962 1.85222 L
.56962 1.87597 L
.55696 1.87597 L
F
.53165 1.85222 m
.5443 1.85222 L
.5443 1.87597 L
.53165 1.87597 L
F
.34177 1.85222 m
.35443 1.85222 L
.35443 1.87597 L
.34177 1.87597 L
F
.59494 1.85222 m
.60759 1.85222 L
.60759 1.87597 L
.59494 1.87597 L
F
.5443 1.85222 m
.55696 1.85222 L
.55696 1.87597 L
.5443 1.87597 L
F
.29114 1.85222 m
.3038 1.85222 L
.3038 1.87597 L
.29114 1.87597 L
F
.3038 1.85222 m
.31646 1.85222 L
.31646 1.87597 L
.3038 1.87597 L
F
.31646 1.85222 m
.32911 1.85222 L
.32911 1.87597 L
.31646 1.87597 L
F
.32911 1.85222 m
.34177 1.85222 L
.34177 1.87597 L
.32911 1.87597 L
F
.35443 1.85222 m
.36709 1.85222 L
.36709 1.87597 L
.35443 1.87597 L
F
.50633 1.85222 m
.51899 1.85222 L
.51899 1.87597 L
.50633 1.87597 L
F
.51899 1.85222 m
.53165 1.85222 L
.53165 1.87597 L
.51899 1.87597 L
F
.4557 1.85222 m
.46835 1.85222 L
.46835 1.87597 L
.4557 1.87597 L
F
.49367 1.85222 m
.50633 1.85222 L
.50633 1.87597 L
.49367 1.87597 L
F
.48101 1.85222 m
.49367 1.85222 L
.49367 1.87597 L
.48101 1.87597 L
F
.58228 1.85222 m
.59494 1.85222 L
.59494 1.87597 L
.58228 1.87597 L
F
.56962 1.85222 m
.58228 1.85222 L
.58228 1.87597 L
.56962 1.87597 L
F
.44304 1.85222 m
.4557 1.85222 L
.4557 1.87597 L
.44304 1.87597 L
F
.46835 1.85222 m
.48101 1.85222 L
.48101 1.87597 L
.46835 1.87597 L
F
.53165 1.82847 m
.5443 1.82847 L
.5443 1.85222 L
.53165 1.85222 L
F
.50633 1.82847 m
.51899 1.82847 L
.51899 1.85222 L
.50633 1.85222 L
F
.64557 1.82847 m
.65823 1.82847 L
.65823 1.85222 L
.64557 1.85222 L
F
.56962 1.82847 m
.58228 1.82847 L
.58228 1.85222 L
.56962 1.85222 L
F
.51899 1.82847 m
.53165 1.82847 L
.53165 1.85222 L
.51899 1.85222 L
F
.60759 1.82847 m
.62025 1.82847 L
.62025 1.85222 L
.60759 1.85222 L
F
.59494 1.82847 m
.60759 1.82847 L
.60759 1.85222 L
.59494 1.85222 L
F
.63291 1.82847 m
.64557 1.82847 L
.64557 1.85222 L
.63291 1.85222 L
F
.65823 1.82847 m
.67089 1.82847 L
.67089 1.85222 L
.65823 1.85222 L
F
.37975 1.85222 m
.39241 1.85222 L
.39241 1.87597 L
.37975 1.87597 L
F
.43038 1.85222 m
.44304 1.85222 L
.44304 1.87597 L
.43038 1.87597 L
F
.36709 1.85222 m
.37975 1.85222 L
.37975 1.87597 L
.36709 1.87597 L
F
.41772 1.85222 m
.43038 1.85222 L
.43038 1.87597 L
.41772 1.87597 L
F
.40506 1.85222 m
.41772 1.85222 L
.41772 1.87597 L
.40506 1.87597 L
F
.5443 1.82847 m
.55696 1.82847 L
.55696 1.85222 L
.5443 1.85222 L
F
.58228 1.82847 m
.59494 1.82847 L
.59494 1.85222 L
.58228 1.85222 L
F
.39241 1.85222 m
.40506 1.85222 L
.40506 1.87597 L
.39241 1.87597 L
F
.55696 1.82847 m
.56962 1.82847 L
.56962 1.85222 L
.55696 1.85222 L
F
.17722 1.85222 m
.18987 1.85222 L
.18987 1.87597 L
.17722 1.87597 L
F
.50633 1.80473 m
.51899 1.80473 L
.51899 1.82847 L
.50633 1.82847 L
F
.16456 1.85222 m
.17722 1.85222 L
.17722 1.87597 L
.16456 1.87597 L
F
.49367 1.80473 m
.50633 1.80473 L
.50633 1.82847 L
.49367 1.82847 L
F
.48101 1.80473 m
.49367 1.80473 L
.49367 1.82847 L
.48101 1.82847 L
F
.12658 1.85222 m
.13924 1.85222 L
.13924 1.87597 L
.12658 1.87597 L
F
.13924 1.85222 m
.1519 1.85222 L
.1519 1.87597 L
.13924 1.87597 L
F
.1519 1.85222 m
.16456 1.85222 L
.16456 1.87597 L
.1519 1.87597 L
F
.20253 1.85222 m
.21519 1.85222 L
.21519 1.87597 L
.20253 1.87597 L
F
.18987 1.85222 m
.20253 1.85222 L
.20253 1.87597 L
.18987 1.87597 L
F
.41772 1.80473 m
.43038 1.80473 L
.43038 1.82847 L
.41772 1.82847 L
F
.46835 1.80473 m
.48101 1.80473 L
.48101 1.82847 L
.46835 1.82847 L
F
.40506 1.80473 m
.41772 1.80473 L
.41772 1.82847 L
.40506 1.82847 L
F
.4557 1.80473 m
.46835 1.80473 L
.46835 1.82847 L
.4557 1.82847 L
F
.44304 1.80473 m
.4557 1.80473 L
.4557 1.82847 L
.44304 1.82847 L
F
.51899 1.80473 m
.53165 1.80473 L
.53165 1.82847 L
.51899 1.82847 L
F
.5443 1.80473 m
.55696 1.80473 L
.55696 1.82847 L
.5443 1.82847 L
F
.43038 1.80473 m
.44304 1.80473 L
.44304 1.82847 L
.43038 1.82847 L
F
.53165 1.80473 m
.5443 1.80473 L
.5443 1.82847 L
.53165 1.82847 L
F
.10127 1.85222 m
.11392 1.85222 L
.11392 1.87597 L
.10127 1.87597 L
F
0 1.85222 m
.01266 1.85222 L
.01266 1.87597 L
0 1.87597 L
F
.08861 1.85222 m
.10127 1.85222 L
.10127 1.87597 L
.08861 1.87597 L
F
.98734 1.82847 m
1 1.82847 L
1 1.85222 L
.98734 1.85222 L
F
.97468 1.82847 m
.98734 1.82847 L
.98734 1.85222 L
.97468 1.85222 L
F
.05063 1.85222 m
.06329 1.85222 L
.06329 1.87597 L
.05063 1.87597 L
F
.07595 1.85222 m
.08861 1.85222 L
.08861 1.87597 L
.07595 1.87597 L
F
.11392 1.85222 m
.12658 1.85222 L
.12658 1.87597 L
.11392 1.87597 L
F
.06329 1.85222 m
.07595 1.85222 L
.07595 1.87597 L
.06329 1.87597 L
F
.22785 1.85222 m
.24051 1.85222 L
.24051 1.87597 L
.22785 1.87597 L
F
.27848 1.85222 m
.29114 1.85222 L
.29114 1.87597 L
.27848 1.87597 L
F
.21519 1.85222 m
.22785 1.85222 L
.22785 1.87597 L
.21519 1.87597 L
F
.26582 1.85222 m
.27848 1.85222 L
.27848 1.87597 L
.26582 1.87597 L
F
.25316 1.85222 m
.26582 1.85222 L
.26582 1.87597 L
.25316 1.87597 L
F
.01266 1.85222 m
.02532 1.85222 L
.02532 1.87597 L
.01266 1.87597 L
F
.03797 1.85222 m
.05063 1.85222 L
.05063 1.87597 L
.03797 1.87597 L
F
.24051 1.85222 m
.25316 1.85222 L
.25316 1.87597 L
.24051 1.87597 L
F
.02532 1.85222 m
.03797 1.85222 L
.03797 1.87597 L
.02532 1.87597 L
F
.79114 1.55539 m
.79114 1.56726 L
.78481 1.56726 L
.78481 1.55539 L
F
.44304 .44032 m
.43038 .43841 L
.43038 .42744 L
.44304 .42744 L
F
.79114 1.55539 m
.79747 1.55539 L
.79747 1.56726 L
.79114 1.56726 L
F
.43038 .43841 m
.41772 .43737 L
.41772 .42744 L
.43038 .42744 L
F
.41772 .43737 m
.40506 .43724 L
.40506 .42744 L
.41772 .42744 L
F
.75949 .31248 m
.75316 .3125 L
.75316 .3087 L
.75949 .3087 L
F
.77848 1.55539 m
.78481 1.55539 L
.78481 1.56726 L
.77848 1.56726 L
F
.74684 .31377 m
.74051 .31629 L
.74051 .3087 L
.74684 .3087 L
F
.76582 1.56223 m
.76582 1.56726 L
.75949 1.56726 L
.75949 1.56349 L
F
.77215 1.55956 m
.77215 1.56726 L
.76582 1.56726 L
.76582 1.56223 L
F
.40506 .43724 m
.39241 .43806 L
.39241 .42744 L
.40506 .42744 L
F
.36709 .44252 m
.35443 .44614 L
.35443 .42744 L
.36709 .42744 L
F
.39241 .43806 m
.37975 .43982 L
.37975 .42744 L
.39241 .42744 L
F
.71519 1.5079 m
.70886 1.5079 L
.70886 1.49602 L
.71519 1.49602 L
F
.37975 .43982 m
.36709 .44252 L
.36709 .42744 L
.37975 .42744 L
F
.4557 .44306 m
.44304 .44032 L
.44304 .42744 L
.4557 .42744 L
F
.48101 .45077 m
.46835 .44656 L
.46835 .42744 L
.48101 .42744 L
F
.35443 .44614 m
.34177 .45061 L
.34177 .42744 L
.35443 .42744 L
F
.46835 .44656 m
.4557 .44306 L
.4557 .42744 L
.46835 .42744 L
F
.8481 1.85222 m
.86076 1.85222 L
.86076 1.87597 L
.8481 1.87597 L
F
.86076 1.85222 m
.87342 1.85222 L
.87342 1.87597 L
.86076 1.87597 L
F
.87342 1.85222 m
.88608 1.85222 L
.88608 1.87597 L
.87342 1.87597 L
F
.88608 1.85222 m
.89873 1.85222 L
.89873 1.87597 L
.88608 1.87597 L
F
.91139 1.85222 m
.92405 1.85222 L
.92405 1.87597 L
.91139 1.87597 L
F
.67089 1.85222 m
.68354 1.85222 L
.68354 1.87597 L
.67089 1.87597 L
F
.62025 1.85222 m
.63291 1.85222 L
.63291 1.87597 L
.62025 1.87597 L
F
.65823 1.85222 m
.67089 1.85222 L
.67089 1.87597 L
.65823 1.87597 L
F
.64557 1.85222 m
.65823 1.85222 L
.65823 1.87597 L
.64557 1.87597 L
F
.83544 1.85222 m
.8481 1.85222 L
.8481 1.87597 L
.83544 1.87597 L
F
.79747 1.85222 m
.81013 1.85222 L
.81013 1.87597 L
.79747 1.87597 L
F
.82278 1.85222 m
.83544 1.85222 L
.83544 1.87597 L
.82278 1.87597 L
F
.77215 .31641 m
.76582 .31374 L
.76582 .3087 L
.77215 .3087 L
F
.81013 1.85222 m
.82278 1.85222 L
.82278 1.87597 L
.81013 1.87597 L
F
.78481 1.85222 m
.79747 1.85222 L
.79747 1.87597 L
.78481 1.87597 L
F
.89873 1.85222 m
.91139 1.85222 L
.91139 1.87597 L
.89873 1.87597 L
F
.77215 1.85222 m
.78481 1.85222 L
.78481 1.87597 L
.77215 1.87597 L
F
.75949 1.85222 m
.77215 1.85222 L
.77215 1.87597 L
.75949 1.87597 L
F
.67089 1.70974 m
.68354 1.70974 L
.68354 1.73349 L
.67089 1.73349 L
F
.64557 1.70974 m
.65823 1.70974 L
.65823 1.73349 L
.64557 1.73349 L
F
.62025 1.70974 m
.63291 1.70974 L
.63291 1.73349 L
.62025 1.73349 L
F
.87342 1.70974 m
.88608 1.70974 L
.88608 1.73349 L
.87342 1.73349 L
F
.65823 1.70974 m
.67089 1.70974 L
.67089 1.73349 L
.65823 1.73349 L
F
.72152 1.70974 m
.73418 1.70974 L
.73418 1.73349 L
.72152 1.73349 L
F
.73418 1.70974 m
.74684 1.70974 L
.74684 1.73349 L
.73418 1.73349 L
F
.74684 1.70974 m
.75949 1.70974 L
.75949 1.73349 L
.74684 1.73349 L
F
.60759 1.70974 m
.62025 1.70974 L
.62025 1.73349 L
.60759 1.73349 L
F
.63291 1.70974 m
.64557 1.70974 L
.64557 1.73349 L
.63291 1.73349 L
F
.78481 1.70974 m
.79747 1.70974 L
.79747 1.73349 L
.78481 1.73349 L
F
.75949 1.70974 m
.77215 1.70974 L
.77215 1.73349 L
.75949 1.73349 L
F
.89873 1.70974 m
.91139 1.70974 L
.91139 1.73349 L
.89873 1.73349 L
F
.82278 1.70974 m
.83544 1.70974 L
.83544 1.73349 L
.82278 1.73349 L
F
.77215 1.70974 m
.78481 1.70974 L
.78481 1.73349 L
.77215 1.73349 L
F
.86076 1.70974 m
.87342 1.70974 L
.87342 1.73349 L
.86076 1.73349 L
F
.8481 1.70974 m
.86076 1.70974 L
.86076 1.73349 L
.8481 1.73349 L
F
.88608 1.70974 m
.89873 1.70974 L
.89873 1.73349 L
.88608 1.73349 L
F
.91139 1.70974 m
.92405 1.70974 L
.92405 1.73349 L
.91139 1.73349 L
F
.72785 1.55539 m
.72785 1.56726 L
.72152 1.56726 L
.72152 1.55539 L
F
.75316 1.56347 m
.75316 1.56726 L
.74684 1.56726 L
.74684 1.5622 L
F
.71519 1.55539 m
.70886 1.55539 L
.70886 1.54352 L
.71519 1.54352 L
F
.75949 1.56349 m
.75949 1.56726 L
.75316 1.56726 L
.75316 1.56347 L
F
.74051 1.55968 m
.74051 1.56726 L
.73418 1.56726 L
.73418 1.55593 L
F
.71519 1.5079 m
.71519 1.51977 L
.70886 1.51977 L
.70886 1.5079 L
F
.77215 .3766 m
.77215 .37994 L
.76582 .37994 L
.76582 .37984 L
F
.74684 1.5622 m
.74684 1.56726 L
.74051 1.56726 L
.74051 1.55968 L
F
.77848 .37141 m
.77848 .37994 L
.77215 .37994 L
.77215 .3766 L
F
.79114 1.53164 m
.79747 1.53164 L
.79747 1.54352 L
.79114 1.54352 L
F
.70886 1.70974 m
.72152 1.70974 L
.72152 1.73349 L
.70886 1.73349 L
F
.71519 1.55539 m
.71519 1.56726 L
.70886 1.56726 L
.70886 1.55539 L
F
.6962 1.70974 m
.70886 1.70974 L
.70886 1.73349 L
.6962 1.73349 L
F
.68354 1.70974 m
.6962 1.70974 L
.6962 1.73349 L
.68354 1.73349 L
F
.77215 1.49937 m
.77215 1.49602 L
.77848 1.49602 L
.77848 1.50456 L
F
.71519 1.53164 m
.70886 1.53164 L
.70886 1.51977 L
.71519 1.51977 L
F
.71519 1.55539 m
.72152 1.55539 L
.72152 1.56726 L
.71519 1.56726 L
F
.71519 1.53164 m
.71519 1.54352 L
.70886 1.54352 L
.70886 1.53164 L
F
.32911 .45709 m
.31646 .46457 L
.31646 .45118 L
.32911 .45118 L
F
.22785 .55076 m
.21519 .5662 L
.21519 .54617 L
.22785 .54617 L
F
.50633 .46364 m
.49367 .45678 L
.49367 .45118 L
.50633 .45118 L
F
.71984 1.51977 m
.71976 1.53164 L
.71519 1.53164 L
.71519 1.51977 L
F
.26582 .5071 m
.25316 .52048 L
.25316 .49867 L
.26582 .49867 L
F
.51899 .4712 m
.50633 .46364 L
.50633 .45118 L
.51899 .45118 L
F
.5443 .49236 m
.53165 .48079 L
.53165 .47493 L
.5443 .47493 L
F
.08576 .75989 m
.07841 .78363 L
.07595 .78363 L
.07595 .75989 L
F
.31646 .46457 m
.3038 .47264 L
.3038 .45118 L
.31646 .45118 L
F
.29114 .48305 m
.27848 .49417 L
.27848 .47493 L
.29114 .47493 L
F
.78771 .3562 m
.7894 .34432 L
.79114 .34432 L
.79114 .3562 L
F
.71976 .34432 m
.71984 .3562 L
.71519 .3562 L
.71519 .34432 L
F
.7894 1.53164 m
.78771 1.51977 L
.79114 1.51977 L
.79114 1.53164 L
F
.7894 .34432 m
.78682 .33245 L
.79114 .33245 L
.79114 .34432 L
F
.78682 1.54352 m
.7894 1.53164 L
.79114 1.53164 L
.79114 1.54352 L
F
.73418 1.49824 m
.72785 1.5036 L
.72785 1.49602 L
.73418 1.49602 L
F
.78481 .32911 m
.77848 .32061 L
.77848 .32058 L
.78481 .32058 L
F
.77215 1.49937 m
.76582 1.49613 L
.76582 1.49602 L
.77215 1.49602 L
F
.75949 1.49394 m
.74684 1.49343 L
.74684 1.47228 L
.75949 1.47228 L
F
.01266 1.82847 m
.02532 1.82847 L
.02532 1.85222 L
.01266 1.85222 L
F
.89873 1.80473 m
.91139 1.80473 L
.91139 1.82847 L
.89873 1.82847 L
F
0 1.82847 m
.01266 1.82847 L
.01266 1.85222 L
0 1.85222 L
F
.88608 1.80473 m
.89873 1.80473 L
.89873 1.82847 L
.88608 1.82847 L
F
.87342 1.80473 m
.88608 1.80473 L
.88608 1.82847 L
.87342 1.82847 L
F
.96203 1.80473 m
.97468 1.80473 L
.97468 1.82847 L
.96203 1.82847 L
F
.98734 1.80473 m
1 1.80473 L
1 1.82847 L
.98734 1.82847 L
F
.02532 1.82847 m
.03797 1.82847 L
.03797 1.85222 L
.02532 1.85222 L
F
.97468 1.80473 m
.98734 1.80473 L
.98734 1.82847 L
.97468 1.82847 L
F
.18987 .59681 m
.17722 .61219 L
.17722 .59366 L
.18987 .59366 L
F
.07284 .80738 m
.06876 .83112 L
.06329 .83112 L
.06329 .80738 L
F
.1519 .64342 m
.13924 .66036 L
.13924 .64115 L
.1519 .64115 L
F
.06581 .85487 m
.06382 .87862 L
.06329 .87862 L
.06329 .85487 L
F
.06876 .83112 m
.06581 .85487 L
.06329 .85487 L
.06329 .83112 L
F
.94937 1.80473 m
.96203 1.80473 L
.96203 1.82847 L
.94937 1.82847 L
F
.93671 1.80473 m
.94937 1.80473 L
.94937 1.82847 L
.93671 1.82847 L
F
.92405 1.80473 m
.93671 1.80473 L
.93671 1.82847 L
.92405 1.82847 L
F
.91139 1.80473 m
.92405 1.80473 L
.92405 1.82847 L
.91139 1.82847 L
F
.75949 .31248 m
.75949 .3087 L
.76582 .3087 L
.76582 .31374 L
F
.79114 .32058 m
.79747 .32058 L
.79747 .33245 L
.79114 .33245 L
F
.79114 .32058 m
.78481 .32058 L
.78481 .3087 L
.79114 .3087 L
F
.71519 .34432 m
.71519 .3562 L
.70886 .3562 L
.70886 .34432 L
F
.79114 .34432 m
.79747 .34432 L
.79747 .3562 L
.79114 .3562 L
F
.71519 .36807 m
.70886 .36807 L
.70886 .3562 L
.71519 .3562 L
F
.71519 .36807 m
.72152 .36807 L
.72152 .37994 L
.71519 .37994 L
F
.97468 1.85222 m
.98734 1.85222 L
.98734 1.87597 L
.97468 1.87597 L
F
.73418 .37773 m
.73418 .37994 L
.72785 .37994 L
.72785 .37236 L
F
.71519 .36807 m
.71519 .37994 L
.70886 .37994 L
.70886 .36807 L
F
.72152 1.85222 m
.73418 1.85222 L
.73418 1.87597 L
.72152 1.87597 L
F
.73418 1.85222 m
.74684 1.85222 L
.74684 1.87597 L
.73418 1.87597 L
F
.74684 1.85222 m
.75949 1.85222 L
.75949 1.87597 L
.74684 1.87597 L
F
.60759 1.85222 m
.62025 1.85222 L
.62025 1.87597 L
.60759 1.87597 L
F
.63291 1.85222 m
.64557 1.85222 L
.64557 1.87597 L
.63291 1.87597 L
F
.70886 1.85222 m
.72152 1.85222 L
.72152 1.87597 L
.70886 1.87597 L
F
.71519 .34432 m
.70886 .34432 L
.70886 .33245 L
.71519 .33245 L
F
.6962 1.85222 m
.70886 1.85222 L
.70886 1.87597 L
.6962 1.87597 L
F
.68354 1.85222 m
.6962 1.85222 L
.6962 1.87597 L
.68354 1.87597 L
F
.06382 .99735 m
.06581 1.0211 L
.06329 1.0211 L
.06329 .99735 L
F
.06221 .94986 m
.0627 .9736 L
.05063 .9736 L
.05063 .94986 L
F
.0627 .90236 m
.06221 .92611 L
.05063 .92611 L
.05063 .90236 L
F
.71519 .32058 m
.72152 .32058 L
.72152 .33245 L
.71519 .33245 L
F
.71519 .32058 m
.70886 .32058 L
.70886 .3087 L
.71519 .3087 L
F
.06876 1.04484 m
.07284 1.06859 L
.06329 1.06859 L
.06329 1.04484 L
F
.06581 1.0211 m
.06876 1.04484 L
.06329 1.04484 L
.06329 1.0211 L
F
.06221 .92611 m
.06221 .94986 L
.05063 .94986 L
.05063 .92611 L
F
.07841 1.09233 m
.08576 1.11608 L
.07595 1.11608 L
.07595 1.09233 L
F
.92405 1.85222 m
.93671 1.85222 L
.93671 1.87597 L
.92405 1.87597 L
F
.93671 1.85222 m
.94937 1.85222 L
.94937 1.87597 L
.93671 1.87597 L
F
.94937 1.85222 m
.96203 1.85222 L
.96203 1.87597 L
.94937 1.87597 L
F
.96203 1.85222 m
.97468 1.85222 L
.97468 1.87597 L
.96203 1.87597 L
F
.98734 1.85222 m
1 1.85222 L
1 1.87597 L
.98734 1.87597 L
F
.74684 .31377 m
.74684 .3087 L
.75316 .3087 L
.75316 .3125 L
F
.71519 .32058 m
.71519 .33245 L
.70886 .33245 L
.70886 .32058 L
F
.73418 .32003 m
.73418 .3087 L
.74051 .3087 L
.74051 .31629 L
F
.72785 .32058 m
.72152 .32058 L
.72152 .3087 L
.72785 .3087 L
F
.88608 1.28231 m
.89873 1.28231 L
.89873 1.30605 L
.88608 1.30605 L
F
.91139 1.28231 m
.92405 1.28231 L
.92405 1.30605 L
.91139 1.30605 L
F
.83544 1.21107 m
.8481 1.21107 L
.8481 1.23481 L
.83544 1.23481 L
F
.8481 1.28231 m
.86076 1.28231 L
.86076 1.30605 L
.8481 1.30605 L
F
.89873 1.28231 m
.91139 1.28231 L
.91139 1.30605 L
.89873 1.30605 L
F
.87342 1.21107 m
.88608 1.21107 L
.88608 1.23481 L
.87342 1.23481 L
F
.82278 1.21107 m
.83544 1.21107 L
.83544 1.23481 L
.82278 1.23481 L
F
.88608 1.21107 m
.89873 1.21107 L
.89873 1.23481 L
.88608 1.23481 L
F
.8481 1.21107 m
.86076 1.21107 L
.86076 1.23481 L
.8481 1.23481 L
F
.81013 1.21107 m
.82278 1.21107 L
.82278 1.23481 L
.81013 1.23481 L
F
.93671 1.28231 m
.94937 1.28231 L
.94937 1.30605 L
.93671 1.30605 L
F
.92405 1.28231 m
.93671 1.28231 L
.93671 1.30605 L
.92405 1.30605 L
F
.98734 1.28231 m
1 1.28231 L
1 1.30605 L
.98734 1.30605 L
F
.94937 1.28231 m
.96203 1.28231 L
.96203 1.30605 L
.94937 1.30605 L
F
.96203 1.28231 m
.97468 1.28231 L
.97468 1.30605 L
.96203 1.30605 L
F
.86076 1.28231 m
.87342 1.28231 L
.87342 1.30605 L
.86076 1.30605 L
F
.87342 1.28231 m
.88608 1.28231 L
.88608 1.30605 L
.87342 1.30605 L
F
0 1.30605 m
.01266 1.30605 L
.01266 1.3298 L
0 1.3298 L
F
.97468 1.28231 m
.98734 1.28231 L
.98734 1.30605 L
.97468 1.30605 L
F
.05063 1.23481 m
.06329 1.23481 L
.06329 1.25856 L
.05063 1.25856 L
F
0 1.23481 m
.01266 1.23481 L
.01266 1.25856 L
0 1.25856 L
F
.06329 1.23481 m
.07595 1.23481 L
.07595 1.25856 L
.06329 1.25856 L
F
.02532 1.23481 m
.03797 1.23481 L
.03797 1.25856 L
.02532 1.25856 L
F
.98734 1.21107 m
1 1.21107 L
1 1.23481 L
.98734 1.23481 L
F
.89873 1.21107 m
.91139 1.21107 L
.91139 1.23481 L
.89873 1.23481 L
F
.93671 1.21107 m
.94937 1.21107 L
.94937 1.23481 L
.93671 1.23481 L
F
.91139 1.21107 m
.92405 1.21107 L
.92405 1.23481 L
.91139 1.23481 L
F
.03797 1.23481 m
.05063 1.23481 L
.05063 1.25856 L
.03797 1.25856 L
F
.92405 1.21107 m
.93671 1.21107 L
.93671 1.23481 L
.92405 1.23481 L
F
.72152 1.21107 m
.73418 1.21107 L
.73418 1.23481 L
.72152 1.23481 L
F
.75949 1.21107 m
.77215 1.21107 L
.77215 1.23481 L
.75949 1.23481 L
F
.73418 1.21107 m
.74684 1.21107 L
.74684 1.23481 L
.73418 1.23481 L
F
.86076 1.21107 m
.87342 1.21107 L
.87342 1.23481 L
.86076 1.23481 L
F
.74684 1.21107 m
.75949 1.21107 L
.75949 1.23481 L
.74684 1.23481 L
F
.77215 1.21107 m
.78481 1.21107 L
.78481 1.23481 L
.77215 1.23481 L
F
.01266 1.23481 m
.02532 1.23481 L
.02532 1.25856 L
.01266 1.25856 L
F
.78481 1.21107 m
.79747 1.21107 L
.79747 1.23481 L
.78481 1.23481 L
F
.79747 1.21107 m
.81013 1.21107 L
.81013 1.23481 L
.79747 1.23481 L
F
.64557 1.30605 m
.65823 1.30605 L
.65823 1.3298 L
.64557 1.3298 L
F
.06329 1.30605 m
.07595 1.30605 L
.07595 1.3298 L
.06329 1.3298 L
F
.65823 1.30605 m
.67089 1.30605 L
.67089 1.3298 L
.65823 1.3298 L
F
.07595 1.30605 m
.08861 1.30605 L
.08861 1.3298 L
.07595 1.3298 L
F
.08861 1.30605 m
.10127 1.30605 L
.10127 1.3298 L
.08861 1.3298 L
F
.6962 1.30605 m
.70886 1.30605 L
.70886 1.3298 L
.6962 1.3298 L
F
.68354 1.30605 m
.6962 1.30605 L
.6962 1.3298 L
.68354 1.3298 L
F
.67089 1.30605 m
.68354 1.30605 L
.68354 1.3298 L
.67089 1.3298 L
F
.62025 1.30605 m
.63291 1.30605 L
.63291 1.3298 L
.62025 1.3298 L
F
.63291 1.30605 m
.64557 1.30605 L
.64557 1.3298 L
.63291 1.3298 L
F
.17722 1.30605 m
.18987 1.30605 L
.18987 1.3298 L
.17722 1.3298 L
F
.16456 1.30605 m
.17722 1.30605 L
.17722 1.3298 L
.16456 1.3298 L
F
.1519 1.30605 m
.16456 1.30605 L
.16456 1.3298 L
.1519 1.3298 L
F
.10127 1.30605 m
.11392 1.30605 L
.11392 1.3298 L
.10127 1.3298 L
F
.11392 1.30605 m
.12658 1.30605 L
.12658 1.3298 L
.11392 1.3298 L
F
.01266 1.30605 m
.02532 1.30605 L
.02532 1.3298 L
.01266 1.3298 L
F
.02532 1.30605 m
.03797 1.30605 L
.03797 1.3298 L
.02532 1.3298 L
F
.03797 1.30605 m
.05063 1.30605 L
.05063 1.3298 L
.03797 1.3298 L
F
.05063 1.30605 m
.06329 1.30605 L
.06329 1.3298 L
.05063 1.3298 L
F
.70886 1.28231 m
.72152 1.28231 L
.72152 1.30605 L
.70886 1.30605 L
F
.6962 1.28231 m
.70886 1.28231 L
.70886 1.30605 L
.6962 1.30605 L
F
.67089 1.28231 m
.68354 1.28231 L
.68354 1.30605 L
.67089 1.30605 L
F
.83544 1.28231 m
.8481 1.28231 L
.8481 1.30605 L
.83544 1.30605 L
F
.81013 1.28231 m
.82278 1.28231 L
.82278 1.30605 L
.81013 1.30605 L
F
.74684 1.28231 m
.75949 1.28231 L
.75949 1.30605 L
.74684 1.30605 L
F
.72152 1.28231 m
.73418 1.28231 L
.73418 1.30605 L
.72152 1.30605 L
F
.68354 1.28231 m
.6962 1.28231 L
.6962 1.30605 L
.68354 1.30605 L
F
.73418 1.28231 m
.74684 1.28231 L
.74684 1.30605 L
.73418 1.30605 L
F
.60759 1.30605 m
.62025 1.30605 L
.62025 1.3298 L
.60759 1.3298 L
F
.59494 1.30605 m
.60759 1.30605 L
.60759 1.3298 L
.59494 1.3298 L
F
.78481 1.28231 m
.79747 1.28231 L
.79747 1.30605 L
.78481 1.30605 L
F
.18987 1.30605 m
.20253 1.30605 L
.20253 1.3298 L
.18987 1.3298 L
F
.22785 1.32521 m
.22785 1.3298 L
.21519 1.3298 L
.21519 1.30976 L
F
.77215 1.28231 m
.78481 1.28231 L
.78481 1.30605 L
.77215 1.30605 L
F
.82278 1.28231 m
.83544 1.28231 L
.83544 1.30605 L
.82278 1.30605 L
F
.79747 1.28231 m
.81013 1.28231 L
.81013 1.30605 L
.79747 1.30605 L
F
.75949 1.28231 m
.77215 1.28231 L
.77215 1.30605 L
.75949 1.30605 L
F
.03797 1.25856 m
.05063 1.25856 L
.05063 1.28231 L
.03797 1.28231 L
F
.1519 1.25856 m
.16456 1.25856 L
.16456 1.28231 L
.1519 1.28231 L
F
.05063 1.25856 m
.06329 1.25856 L
.06329 1.28231 L
.05063 1.28231 L
F
.11392 1.25856 m
.12658 1.25856 L
.12658 1.28231 L
.11392 1.28231 L
F
.18987 1.27916 m
.18987 1.28231 L
.17722 1.28231 L
.17722 1.26378 L
F
.08861 1.25856 m
.10127 1.25856 L
.10127 1.28231 L
.08861 1.28231 L
F
.07595 1.25856 m
.08861 1.25856 L
.08861 1.28231 L
.07595 1.28231 L
F
.06329 1.25856 m
.07595 1.25856 L
.07595 1.28231 L
.06329 1.28231 L
F
.01266 1.25856 m
.02532 1.25856 L
.02532 1.28231 L
.01266 1.28231 L
F
.02532 1.25856 m
.03797 1.25856 L
.03797 1.28231 L
.02532 1.28231 L
F
.89873 1.23481 m
.91139 1.23481 L
.91139 1.25856 L
.89873 1.25856 L
F
.8481 1.23481 m
.86076 1.23481 L
.86076 1.25856 L
.8481 1.25856 L
F
.91139 1.23481 m
.92405 1.23481 L
.92405 1.25856 L
.91139 1.25856 L
F
.87342 1.23481 m
.88608 1.23481 L
.88608 1.25856 L
.87342 1.25856 L
F
.83544 1.23481 m
.8481 1.23481 L
.8481 1.25856 L
.83544 1.25856 L
F
.13924 1.25856 m
.1519 1.25856 L
.1519 1.28231 L
.13924 1.28231 L
F
.10127 1.25856 m
.11392 1.25856 L
.11392 1.28231 L
.10127 1.28231 L
F
.88608 1.23481 m
.89873 1.23481 L
.89873 1.25856 L
.88608 1.25856 L
F
.12658 1.25856 m
.13924 1.25856 L
.13924 1.28231 L
.12658 1.28231 L
F
.78481 1.23481 m
.79747 1.23481 L
.79747 1.25856 L
.78481 1.25856 L
F
.77215 1.23481 m
.78481 1.23481 L
.78481 1.25856 L
.77215 1.25856 L
F
.74684 1.23481 m
.75949 1.23481 L
.75949 1.25856 L
.74684 1.25856 L
F
.13924 1.23481 m
.1519 1.23481 L
.1519 1.25856 L
.13924 1.25856 L
F
.12658 1.23481 m
.13924 1.23481 L
.13924 1.25856 L
.12658 1.25856 L
F
.82278 1.23481 m
.83544 1.23481 L
.83544 1.25856 L
.82278 1.25856 L
F
.79747 1.23481 m
.81013 1.23481 L
.81013 1.25856 L
.79747 1.25856 L
F
.75949 1.23481 m
.77215 1.23481 L
.77215 1.25856 L
.75949 1.25856 L
F
.81013 1.23481 m
.82278 1.23481 L
.82278 1.25856 L
.81013 1.25856 L
F
.65823 1.23481 m
.67089 1.23481 L
.67089 1.25856 L
.65823 1.25856 L
F
.64557 1.23481 m
.65823 1.23481 L
.65823 1.25856 L
.64557 1.25856 L
F
.63291 1.23481 m
.64557 1.23481 L
.64557 1.25856 L
.63291 1.25856 L
F
.60759 1.23481 m
.62025 1.23481 L
.62025 1.25856 L
.60759 1.25856 L
F
.62025 1.23481 m
.63291 1.23481 L
.63291 1.25856 L
.62025 1.25856 L
F
.07595 1.23481 m
.08861 1.23481 L
.08861 1.25856 L
.07595 1.25856 L
F
.08861 1.23481 m
.10127 1.23481 L
.10127 1.25856 L
.08861 1.25856 L
F
.10127 1.23481 m
.11392 1.23481 L
.11392 1.25856 L
.10127 1.25856 L
F
.11392 1.23481 m
.12658 1.23481 L
.12658 1.25856 L
.11392 1.25856 L
F
.93671 1.18732 m
.94937 1.18732 L
.94937 1.21107 L
.93671 1.21107 L
F
.97468 1.18732 m
.98734 1.18732 L
.98734 1.21107 L
.97468 1.21107 L
F
.94937 1.18732 m
.96203 1.18732 L
.96203 1.21107 L
.94937 1.21107 L
F
.07595 1.21107 m
.08861 1.21107 L
.08861 1.23481 L
.07595 1.23481 L
F
.96203 1.18732 m
.97468 1.18732 L
.97468 1.21107 L
.96203 1.21107 L
F
.65823 1.21107 m
.67089 1.21107 L
.67089 1.23481 L
.65823 1.23481 L
F
.98734 1.18732 m
1 1.18732 L
1 1.21107 L
.98734 1.21107 L
F
.67089 1.21107 m
.68354 1.21107 L
.68354 1.23481 L
.67089 1.23481 L
F
0 1.21107 m
.01266 1.21107 L
.01266 1.23481 L
0 1.23481 L
F
.01266 1.21107 m
.02532 1.21107 L
.02532 1.23481 L
.01266 1.23481 L
F
.05063 1.21107 m
.06329 1.21107 L
.06329 1.23481 L
.05063 1.23481 L
F
.94937 1.21107 m
.96203 1.21107 L
.96203 1.23481 L
.94937 1.23481 L
F
.06329 1.21107 m
.07595 1.21107 L
.07595 1.23481 L
.06329 1.23481 L
F
.96203 1.21107 m
.97468 1.21107 L
.97468 1.23481 L
.96203 1.23481 L
F
.97468 1.21107 m
.98734 1.21107 L
.98734 1.23481 L
.97468 1.23481 L
F
.08861 1.21107 m
.10127 1.21107 L
.10127 1.23481 L
.08861 1.23481 L
F
.10127 1.21107 m
.11392 1.21107 L
.11392 1.23481 L
.10127 1.23481 L
F
.02532 1.21107 m
.03797 1.21107 L
.03797 1.23481 L
.02532 1.23481 L
F
.03797 1.21107 m
.05063 1.21107 L
.05063 1.23481 L
.03797 1.23481 L
F
.92405 1.23481 m
.93671 1.23481 L
.93671 1.25856 L
.92405 1.25856 L
F
.96203 1.23481 m
.97468 1.23481 L
.97468 1.25856 L
.96203 1.25856 L
F
.93671 1.23481 m
.94937 1.23481 L
.94937 1.25856 L
.93671 1.25856 L
F
.59786 1.21107 m
.60759 1.21107 L
.60759 1.23481 L
.59666 1.23481 L
F
.94937 1.23481 m
.96203 1.23481 L
.96203 1.25856 L
.94937 1.25856 L
F
.97468 1.23481 m
.98734 1.23481 L
.98734 1.25856 L
.97468 1.25856 L
F
.86076 1.23481 m
.87342 1.23481 L
.87342 1.25856 L
.86076 1.25856 L
F
.98734 1.23481 m
1 1.23481 L
1 1.25856 L
.98734 1.25856 L
F
0 1.25856 m
.01266 1.25856 L
.01266 1.28231 L
0 1.28231 L
F
.70886 1.21107 m
.72152 1.21107 L
.72152 1.23481 L
.70886 1.23481 L
F
.6962 1.21107 m
.70886 1.21107 L
.70886 1.23481 L
.6962 1.23481 L
F
.68354 1.21107 m
.6962 1.21107 L
.6962 1.23481 L
.68354 1.23481 L
F
.63291 1.21107 m
.64557 1.21107 L
.64557 1.23481 L
.63291 1.23481 L
F
.64557 1.21107 m
.65823 1.21107 L
.65823 1.23481 L
.64557 1.23481 L
F
.60759 1.21107 m
.62025 1.21107 L
.62025 1.23481 L
.60759 1.23481 L
F
.62025 1.21107 m
.63291 1.21107 L
.63291 1.23481 L
.62025 1.23481 L
F
.1519 1.23254 m
.1519 1.23481 L
.13924 1.23481 L
.13924 1.2156 L
F
.11392 1.21107 m
.12658 1.21107 L
.12658 1.23481 L
.11392 1.23481 L
F
.6962 1.40104 m
.70886 1.40104 L
.70886 1.42478 L
.6962 1.42478 L
F
.67089 1.40104 m
.68354 1.40104 L
.68354 1.42478 L
.67089 1.42478 L
F
.64557 1.40104 m
.65823 1.40104 L
.65823 1.42478 L
.64557 1.42478 L
F
.49367 1.42478 m
.50633 1.42478 L
.50633 1.44853 L
.49367 1.44853 L
F
.68354 1.40104 m
.6962 1.40104 L
.6962 1.42478 L
.68354 1.42478 L
F
.74684 1.40104 m
.75949 1.40104 L
.75949 1.42478 L
.74684 1.42478 L
F
.75949 1.40104 m
.77215 1.40104 L
.77215 1.42478 L
.75949 1.42478 L
F
.77215 1.40104 m
.78481 1.40104 L
.78481 1.42478 L
.77215 1.42478 L
F
.63291 1.40104 m
.64557 1.40104 L
.64557 1.42478 L
.63291 1.42478 L
F
.65823 1.40104 m
.67089 1.40104 L
.67089 1.42478 L
.65823 1.42478 L
F
.40506 1.43872 m
.40506 1.44853 L
.39241 1.44853 L
.39241 1.43791 L
F
.41772 1.4386 m
.41772 1.44853 L
.40506 1.44853 L
.40506 1.43872 L
F
.43038 1.43756 m
.43038 1.44853 L
.41772 1.44853 L
.41772 1.4386 L
F
.44304 1.43565 m
.44304 1.44853 L
.43038 1.44853 L
.43038 1.43756 L
F
.46835 1.42941 m
.46835 1.44853 L
.4557 1.44853 L
.4557 1.43291 L
F
.53165 1.42478 m
.5443 1.42478 L
.5443 1.44853 L
.53165 1.44853 L
F
.48101 1.4252 m
.48101 1.44853 L
.46835 1.44853 L
.46835 1.42941 L
F
.51899 1.42478 m
.53165 1.42478 L
.53165 1.44853 L
.51899 1.44853 L
F
.50633 1.42478 m
.51899 1.42478 L
.51899 1.44853 L
.50633 1.44853 L
F
.58228 1.40104 m
.59494 1.40104 L
.59494 1.42478 L
.58228 1.42478 L
F
.55696 1.40104 m
.56962 1.40104 L
.56962 1.42478 L
.55696 1.42478 L
F
.97468 1.40104 m
.98734 1.40104 L
.98734 1.42478 L
.97468 1.42478 L
F
.62025 1.40104 m
.63291 1.40104 L
.63291 1.42478 L
.62025 1.42478 L
F
.56962 1.40104 m
.58228 1.40104 L
.58228 1.42478 L
.56962 1.42478 L
F
.92405 1.40104 m
.93671 1.40104 L
.93671 1.42478 L
.92405 1.42478 L
F
.93671 1.40104 m
.94937 1.40104 L
.94937 1.42478 L
.93671 1.42478 L
F
.94937 1.40104 m
.96203 1.40104 L
.96203 1.42478 L
.94937 1.42478 L
F
.96203 1.40104 m
.97468 1.40104 L
.97468 1.42478 L
.96203 1.42478 L
F
.98734 1.40104 m
1 1.40104 L
1 1.42478 L
.98734 1.42478 L
F
.53165 1.40104 m
.5443 1.40104 L
.5443 1.42478 L
.53165 1.42478 L
F
.73418 1.40104 m
.74684 1.40104 L
.74684 1.42478 L
.73418 1.42478 L
F
.5443 1.40104 m
.55696 1.40104 L
.55696 1.42478 L
.5443 1.42478 L
F
.72152 1.40104 m
.73418 1.40104 L
.73418 1.42478 L
.72152 1.42478 L
F
.70886 1.40104 m
.72152 1.40104 L
.72152 1.42478 L
.70886 1.42478 L
F
.60759 1.40104 m
.62025 1.40104 L
.62025 1.42478 L
.60759 1.42478 L
F
.59494 1.40104 m
.60759 1.40104 L
.60759 1.42478 L
.59494 1.42478 L
F
.50633 1.41233 m
.50633 1.42478 L
.49367 1.42478 L
.49367 1.41919 L
F
.51899 1.40476 m
.51899 1.42478 L
.50633 1.42478 L
.50633 1.41233 L
F
.12658 1.42478 m
.13924 1.42478 L
.13924 1.44853 L
.12658 1.44853 L
F
.35443 1.42982 m
.35443 1.44853 L
.34177 1.44853 L
.34177 1.42535 L
F
.11392 1.42478 m
.12658 1.42478 L
.12658 1.44853 L
.11392 1.44853 L
F
.31646 1.42478 m
.32911 1.42478 L
.32911 1.44853 L
.31646 1.44853 L
F
.3038 1.42478 m
.31646 1.42478 L
.31646 1.44853 L
.3038 1.44853 L
F
.10127 1.42478 m
.11392 1.42478 L
.11392 1.44853 L
.10127 1.44853 L
F
.07595 1.42478 m
.08861 1.42478 L
.08861 1.44853 L
.07595 1.44853 L
F
.20253 1.42478 m
.21519 1.42478 L
.21519 1.44853 L
.20253 1.44853 L
F
.13924 1.42478 m
.1519 1.42478 L
.1519 1.44853 L
.13924 1.44853 L
F
.08861 1.42478 m
.10127 1.42478 L
.10127 1.44853 L
.08861 1.44853 L
F
.24051 1.42478 m
.25316 1.42478 L
.25316 1.44853 L
.24051 1.44853 L
F
.29114 1.42478 m
.3038 1.42478 L
.3038 1.44853 L
.29114 1.44853 L
F
.22785 1.42478 m
.24051 1.42478 L
.24051 1.44853 L
.22785 1.44853 L
F
.27848 1.42478 m
.29114 1.42478 L
.29114 1.44853 L
.27848 1.44853 L
F
.26582 1.42478 m
.27848 1.42478 L
.27848 1.44853 L
.26582 1.44853 L
F
.36709 1.43344 m
.36709 1.44853 L
.35443 1.44853 L
.35443 1.42982 L
F
.39241 1.43791 m
.39241 1.44853 L
.37975 1.44853 L
.37975 1.43615 L
F
.25316 1.42478 m
.26582 1.42478 L
.26582 1.44853 L
.25316 1.44853 L
F
.37975 1.43615 m
.37975 1.44853 L
.36709 1.44853 L
.36709 1.43344 L
F
.65823 1.42478 m
.67089 1.42478 L
.67089 1.44853 L
.65823 1.44853 L
F
.67089 1.42478 m
.68354 1.42478 L
.68354 1.44853 L
.67089 1.44853 L
F
.68354 1.42478 m
.6962 1.42478 L
.6962 1.44853 L
.68354 1.44853 L
F
.5443 1.42478 m
.55696 1.42478 L
.55696 1.44853 L
.5443 1.44853 L
F
.56962 1.42478 m
.58228 1.42478 L
.58228 1.44853 L
.56962 1.44853 L
F
.64557 1.42478 m
.65823 1.42478 L
.65823 1.44853 L
.64557 1.44853 L
F
.4557 1.43291 m
.4557 1.44853 L
.44304 1.44853 L
.44304 1.43565 L
F
.63291 1.42478 m
.64557 1.42478 L
.64557 1.44853 L
.63291 1.44853 L
F
.62025 1.42478 m
.63291 1.42478 L
.63291 1.44853 L
.62025 1.44853 L
F
.1519 1.42478 m
.16456 1.42478 L
.16456 1.44853 L
.1519 1.44853 L
F
.16456 1.42478 m
.17722 1.42478 L
.17722 1.44853 L
.16456 1.44853 L
F
.17722 1.42478 m
.18987 1.42478 L
.18987 1.44853 L
.17722 1.44853 L
F
.18987 1.42478 m
.20253 1.42478 L
.20253 1.44853 L
.18987 1.44853 L
F
.21519 1.42478 m
.22785 1.42478 L
.22785 1.44853 L
.21519 1.44853 L
F
.60759 1.42478 m
.62025 1.42478 L
.62025 1.44853 L
.60759 1.44853 L
F
.55696 1.42478 m
.56962 1.42478 L
.56962 1.44853 L
.55696 1.44853 L
F
.59494 1.42478 m
.60759 1.42478 L
.60759 1.44853 L
.59494 1.44853 L
F
.58228 1.42478 m
.59494 1.42478 L
.59494 1.44853 L
.58228 1.44853 L
F
.74684 1.25856 m
.75949 1.25856 L
.75949 1.28231 L
.74684 1.28231 L
F
.73418 1.25856 m
.74684 1.25856 L
.74684 1.28231 L
.73418 1.28231 L
F
.72152 1.25856 m
.73418 1.25856 L
.73418 1.28231 L
.72152 1.28231 L
F
.67089 1.25856 m
.68354 1.25856 L
.68354 1.28231 L
.67089 1.28231 L
F
.68354 1.25856 m
.6962 1.25856 L
.6962 1.28231 L
.68354 1.28231 L
F
.59494 1.25856 m
.60759 1.25856 L
.60759 1.28231 L
.59494 1.28231 L
F
.59465 1.25856 m
.59494 1.25856 L
.59494 1.28231 L
.59135 1.28231 L
F
.64557 1.25856 m
.65823 1.25856 L
.65823 1.28231 L
.64557 1.28231 L
F
.60759 1.25856 m
.62025 1.25856 L
.62025 1.28231 L
.60759 1.28231 L
F
.62025 1.25856 m
.63291 1.25856 L
.63291 1.28231 L
.62025 1.28231 L
F
.11392 1.28231 m
.12658 1.28231 L
.12658 1.30605 L
.11392 1.30605 L
F
.1519 1.28231 m
.16456 1.28231 L
.16456 1.30605 L
.1519 1.30605 L
F
.12658 1.28231 m
.13924 1.28231 L
.13924 1.30605 L
.12658 1.30605 L
F
.63291 1.28231 m
.64557 1.28231 L
.64557 1.30605 L
.63291 1.30605 L
F
.13924 1.28231 m
.1519 1.28231 L
.1519 1.30605 L
.13924 1.30605 L
F
.6962 1.25856 m
.70886 1.25856 L
.70886 1.28231 L
.6962 1.28231 L
F
.70886 1.25856 m
.72152 1.25856 L
.72152 1.28231 L
.70886 1.28231 L
F
.17722 1.28231 m
.18987 1.28231 L
.18987 1.30605 L
.17722 1.30605 L
F
.16456 1.28231 m
.17722 1.28231 L
.17722 1.30605 L
.16456 1.30605 L
F
.77215 1.25856 m
.78481 1.25856 L
.78481 1.28231 L
.77215 1.28231 L
F
.75949 1.25856 m
.77215 1.25856 L
.77215 1.28231 L
.75949 1.28231 L
F
.82278 1.25856 m
.83544 1.25856 L
.83544 1.28231 L
.82278 1.28231 L
F
.78481 1.25856 m
.79747 1.25856 L
.79747 1.28231 L
.78481 1.28231 L
F
.79747 1.25856 m
.81013 1.25856 L
.81013 1.28231 L
.79747 1.28231 L
F
.12658 1.30605 m
.13924 1.30605 L
.13924 1.3298 L
.12658 1.3298 L
F
.13924 1.30605 m
.1519 1.30605 L
.1519 1.3298 L
.13924 1.3298 L
F
.83544 1.25856 m
.8481 1.25856 L
.8481 1.28231 L
.83544 1.28231 L
F
.81013 1.25856 m
.82278 1.25856 L
.82278 1.28231 L
.81013 1.28231 L
F
.88608 1.25856 m
.89873 1.25856 L
.89873 1.28231 L
.88608 1.28231 L
F
.87342 1.25856 m
.88608 1.25856 L
.88608 1.28231 L
.87342 1.28231 L
F
.8481 1.25856 m
.86076 1.25856 L
.86076 1.28231 L
.8481 1.28231 L
F
.65823 1.25856 m
.67089 1.25856 L
.67089 1.28231 L
.65823 1.28231 L
F
.63291 1.25856 m
.64557 1.25856 L
.64557 1.28231 L
.63291 1.28231 L
F
.92405 1.25856 m
.93671 1.25856 L
.93671 1.28231 L
.92405 1.28231 L
F
.89873 1.25856 m
.91139 1.25856 L
.91139 1.28231 L
.89873 1.28231 L
F
.86076 1.25856 m
.87342 1.25856 L
.87342 1.28231 L
.86076 1.28231 L
F
.91139 1.25856 m
.92405 1.25856 L
.92405 1.28231 L
.91139 1.28231 L
F
.89873 1.40104 m
.91139 1.40104 L
.91139 1.42478 L
.89873 1.42478 L
F
.81013 1.40104 m
.82278 1.40104 L
.82278 1.42478 L
.81013 1.42478 L
F
.88608 1.40104 m
.89873 1.40104 L
.89873 1.42478 L
.88608 1.42478 L
F
.79747 1.40104 m
.81013 1.40104 L
.81013 1.42478 L
.79747 1.42478 L
F
.78481 1.40104 m
.79747 1.40104 L
.79747 1.42478 L
.78481 1.42478 L
F
.05063 1.28231 m
.06329 1.28231 L
.06329 1.30605 L
.05063 1.30605 L
F
.87342 1.40104 m
.88608 1.40104 L
.88608 1.42478 L
.87342 1.42478 L
F
.06329 1.28231 m
.07595 1.28231 L
.07595 1.30605 L
.06329 1.30605 L
F
.91139 1.40104 m
.92405 1.40104 L
.92405 1.42478 L
.91139 1.42478 L
F
.86076 1.40104 m
.87342 1.40104 L
.87342 1.42478 L
.86076 1.42478 L
F
.01266 1.42478 m
.02532 1.42478 L
.02532 1.44853 L
.01266 1.44853 L
F
.06329 1.42478 m
.07595 1.42478 L
.07595 1.44853 L
.06329 1.44853 L
F
0 1.42478 m
.01266 1.42478 L
.01266 1.44853 L
0 1.44853 L
F
.05063 1.42478 m
.06329 1.42478 L
.06329 1.44853 L
.05063 1.44853 L
F
.03797 1.42478 m
.05063 1.42478 L
.05063 1.44853 L
.03797 1.44853 L
F
.82278 1.40104 m
.83544 1.40104 L
.83544 1.42478 L
.82278 1.42478 L
F
.8481 1.40104 m
.86076 1.40104 L
.86076 1.42478 L
.8481 1.42478 L
F
.02532 1.42478 m
.03797 1.42478 L
.03797 1.44853 L
.02532 1.44853 L
F
.83544 1.40104 m
.8481 1.40104 L
.8481 1.42478 L
.83544 1.42478 L
F
.60759 1.28231 m
.62025 1.28231 L
.62025 1.30605 L
.60759 1.30605 L
F
.98734 1.25856 m
1 1.25856 L
1 1.28231 L
.98734 1.28231 L
F
.62025 1.28231 m
.63291 1.28231 L
.63291 1.30605 L
.62025 1.30605 L
F
0 1.28231 m
.01266 1.28231 L
.01266 1.30605 L
0 1.30605 L
F
.01266 1.28231 m
.02532 1.28231 L
.02532 1.30605 L
.01266 1.30605 L
F
.64557 1.28231 m
.65823 1.28231 L
.65823 1.30605 L
.64557 1.30605 L
F
.65823 1.28231 m
.67089 1.28231 L
.67089 1.30605 L
.65823 1.30605 L
F
.59135 1.28231 m
.59494 1.28231 L
.59494 1.30605 L
.58639 1.30605 L
F
.59494 1.28231 m
.60759 1.28231 L
.60759 1.30605 L
.59494 1.30605 L
F
.10127 1.28231 m
.11392 1.28231 L
.11392 1.30605 L
.10127 1.30605 L
F
.08861 1.28231 m
.10127 1.28231 L
.10127 1.30605 L
.08861 1.30605 L
F
.07595 1.28231 m
.08861 1.28231 L
.08861 1.30605 L
.07595 1.30605 L
F
.02532 1.28231 m
.03797 1.28231 L
.03797 1.30605 L
.02532 1.30605 L
F
.03797 1.28231 m
.05063 1.28231 L
.05063 1.30605 L
.03797 1.30605 L
F
.93671 1.25856 m
.94937 1.25856 L
.94937 1.28231 L
.93671 1.28231 L
F
.94937 1.25856 m
.96203 1.25856 L
.96203 1.28231 L
.94937 1.28231 L
F
.96203 1.25856 m
.97468 1.25856 L
.97468 1.28231 L
.96203 1.28231 L
F
.97468 1.25856 m
.98734 1.25856 L
.98734 1.28231 L
.97468 1.28231 L
F
.74684 1.3298 m
.75949 1.3298 L
.75949 1.35355 L
.74684 1.35355 L
F
.86076 1.3298 m
.87342 1.3298 L
.87342 1.35355 L
.86076 1.35355 L
F
.75949 1.3298 m
.77215 1.3298 L
.77215 1.35355 L
.75949 1.35355 L
F
.87342 1.3298 m
.88608 1.3298 L
.88608 1.35355 L
.87342 1.35355 L
F
.88608 1.3298 m
.89873 1.3298 L
.89873 1.35355 L
.88608 1.35355 L
F
.79747 1.3298 m
.81013 1.3298 L
.81013 1.35355 L
.79747 1.35355 L
F
.78481 1.3298 m
.79747 1.3298 L
.79747 1.35355 L
.78481 1.35355 L
F
.77215 1.3298 m
.78481 1.3298 L
.78481 1.35355 L
.77215 1.35355 L
F
.72152 1.3298 m
.73418 1.3298 L
.73418 1.35355 L
.72152 1.35355 L
F
.73418 1.3298 m
.74684 1.3298 L
.74684 1.35355 L
.73418 1.35355 L
F
.59494 1.35355 m
.60759 1.35355 L
.60759 1.37729 L
.59494 1.37729 L
F
.58228 1.35355 m
.59494 1.35355 L
.59494 1.37729 L
.58228 1.37729 L
F
.56962 1.35355 m
.58228 1.35355 L
.58228 1.37729 L
.56962 1.37729 L
F
.65823 1.35355 m
.67089 1.35355 L
.67089 1.37729 L
.65823 1.37729 L
F
.26582 1.36887 m
.26582 1.37729 L
.25316 1.37729 L
.25316 1.35549 L
F
.81013 1.3298 m
.82278 1.3298 L
.82278 1.35355 L
.81013 1.35355 L
F
.82278 1.3298 m
.83544 1.3298 L
.83544 1.35355 L
.82278 1.35355 L
F
.83544 1.3298 m
.8481 1.3298 L
.8481 1.35355 L
.83544 1.35355 L
F
.8481 1.3298 m
.86076 1.3298 L
.86076 1.35355 L
.8481 1.35355 L
F
.93671 1.3298 m
.94937 1.3298 L
.94937 1.35355 L
.93671 1.35355 L
F
.96203 1.3298 m
.97468 1.3298 L
.97468 1.35355 L
.96203 1.35355 L
F
.39241 1.75723 m
.40506 1.75723 L
.40506 1.78098 L
.39241 1.78098 L
F
.89873 1.3298 m
.91139 1.3298 L
.91139 1.35355 L
.89873 1.35355 L
F
.94937 1.3298 m
.96203 1.3298 L
.96203 1.35355 L
.94937 1.35355 L
F
.35443 1.75723 m
.36709 1.75723 L
.36709 1.78098 L
.35443 1.78098 L
F
.40506 1.75723 m
.41772 1.75723 L
.41772 1.78098 L
.40506 1.78098 L
F
.34177 1.75723 m
.35443 1.75723 L
.35443 1.78098 L
.34177 1.78098 L
F
.37975 1.75723 m
.39241 1.75723 L
.39241 1.78098 L
.37975 1.78098 L
F
.41772 1.75723 m
.43038 1.75723 L
.43038 1.78098 L
.41772 1.78098 L
F
.98734 1.3298 m
1 1.3298 L
1 1.35355 L
.98734 1.35355 L
F
.97468 1.3298 m
.98734 1.3298 L
.98734 1.35355 L
.97468 1.35355 L
F
.03797 1.35355 m
.05063 1.35355 L
.05063 1.37729 L
.03797 1.37729 L
F
0 1.35355 m
.01266 1.35355 L
.01266 1.37729 L
0 1.37729 L
F
.01266 1.35355 m
.02532 1.35355 L
.02532 1.37729 L
.01266 1.37729 L
F
.91139 1.3298 m
.92405 1.3298 L
.92405 1.35355 L
.91139 1.35355 L
F
.92405 1.3298 m
.93671 1.3298 L
.93671 1.35355 L
.92405 1.35355 L
F
.05063 1.35355 m
.06329 1.35355 L
.06329 1.37729 L
.05063 1.37729 L
F
.02532 1.35355 m
.03797 1.35355 L
.03797 1.37729 L
.02532 1.37729 L
F
.91139 1.30605 m
.92405 1.30605 L
.92405 1.3298 L
.91139 1.3298 L
F
.89873 1.30605 m
.91139 1.30605 L
.91139 1.3298 L
.89873 1.3298 L
F
.88608 1.30605 m
.89873 1.30605 L
.89873 1.3298 L
.88608 1.3298 L
F
.03797 1.3298 m
.05063 1.3298 L
.05063 1.35355 L
.03797 1.35355 L
F
.01266 1.3298 m
.02532 1.3298 L
.02532 1.35355 L
.01266 1.35355 L
F
.17722 1.35355 m
.18987 1.35355 L
.18987 1.37729 L
.17722 1.37729 L
F
.92405 1.30605 m
.93671 1.30605 L
.93671 1.3298 L
.92405 1.3298 L
F
.18987 1.35355 m
.20253 1.35355 L
.20253 1.37729 L
.18987 1.37729 L
F
.93671 1.30605 m
.94937 1.30605 L
.94937 1.3298 L
.93671 1.3298 L
F
.94937 1.30605 m
.96203 1.30605 L
.96203 1.3298 L
.94937 1.3298 L
F
.75949 1.30605 m
.77215 1.30605 L
.77215 1.3298 L
.75949 1.3298 L
F
.78481 1.30605 m
.79747 1.30605 L
.79747 1.3298 L
.78481 1.3298 L
F
.98734 1.30605 m
1 1.30605 L
1 1.3298 L
.98734 1.3298 L
F
.72152 1.30605 m
.73418 1.30605 L
.73418 1.3298 L
.72152 1.3298 L
F
.77215 1.30605 m
.78481 1.30605 L
.78481 1.3298 L
.77215 1.3298 L
F
.97468 1.30605 m
.98734 1.30605 L
.98734 1.3298 L
.97468 1.3298 L
F
.02532 1.3298 m
.03797 1.3298 L
.03797 1.35355 L
.02532 1.35355 L
F
0 1.3298 m
.01266 1.3298 L
.01266 1.35355 L
0 1.35355 L
F
.96203 1.30605 m
.97468 1.30605 L
.97468 1.3298 L
.96203 1.3298 L
F
.63291 1.35355 m
.64557 1.35355 L
.64557 1.37729 L
.63291 1.37729 L
F
.11392 1.35355 m
.12658 1.35355 L
.12658 1.37729 L
.11392 1.37729 L
F
.64557 1.35355 m
.65823 1.35355 L
.65823 1.37729 L
.64557 1.37729 L
F
.12658 1.35355 m
.13924 1.35355 L
.13924 1.37729 L
.12658 1.37729 L
F
.13924 1.35355 m
.1519 1.35355 L
.1519 1.37729 L
.13924 1.37729 L
F
.67089 1.35355 m
.68354 1.35355 L
.68354 1.37729 L
.67089 1.37729 L
F
.68354 1.35355 m
.6962 1.35355 L
.6962 1.37729 L
.68354 1.37729 L
F
.60759 1.35355 m
.62025 1.35355 L
.62025 1.37729 L
.60759 1.37729 L
F
.62025 1.35355 m
.63291 1.35355 L
.63291 1.37729 L
.62025 1.37729 L
F
.22785 1.35355 m
.24051 1.35355 L
.24051 1.37729 L
.22785 1.37729 L
F
.21519 1.35355 m
.22785 1.35355 L
.22785 1.37729 L
.21519 1.37729 L
F
.20253 1.35355 m
.21519 1.35355 L
.21519 1.37729 L
.20253 1.37729 L
F
.1519 1.35355 m
.16456 1.35355 L
.16456 1.37729 L
.1519 1.37729 L
F
.16456 1.35355 m
.17722 1.35355 L
.17722 1.37729 L
.16456 1.37729 L
F
.06329 1.35355 m
.07595 1.35355 L
.07595 1.37729 L
.06329 1.37729 L
F
.07595 1.35355 m
.08861 1.35355 L
.08861 1.37729 L
.07595 1.37729 L
F
.08861 1.35355 m
.10127 1.35355 L
.10127 1.37729 L
.08861 1.37729 L
F
.10127 1.35355 m
.11392 1.35355 L
.11392 1.37729 L
.10127 1.37729 L
F
.70886 1.73349 m
.72152 1.73349 L
.72152 1.75723 L
.70886 1.75723 L
F
.72152 1.73349 m
.73418 1.73349 L
.73418 1.75723 L
.72152 1.75723 L
F
.73418 1.73349 m
.74684 1.73349 L
.74684 1.75723 L
.73418 1.75723 L
F
.78481 1.73349 m
.79747 1.73349 L
.79747 1.75723 L
.78481 1.75723 L
F
.77215 1.73349 m
.78481 1.73349 L
.78481 1.75723 L
.77215 1.75723 L
F
.81013 1.73349 m
.82278 1.73349 L
.82278 1.75723 L
.81013 1.75723 L
F
.86076 1.73349 m
.87342 1.73349 L
.87342 1.75723 L
.86076 1.75723 L
F
.79747 1.73349 m
.81013 1.73349 L
.81013 1.75723 L
.79747 1.75723 L
F
.8481 1.73349 m
.86076 1.73349 L
.86076 1.75723 L
.8481 1.75723 L
F
.83544 1.73349 m
.8481 1.73349 L
.8481 1.75723 L
.83544 1.75723 L
F
.64557 1.75723 m
.65823 1.75723 L
.65823 1.78098 L
.64557 1.78098 L
F
.62025 1.75723 m
.63291 1.75723 L
.63291 1.78098 L
.62025 1.78098 L
F
.59494 1.75723 m
.60759 1.75723 L
.60759 1.78098 L
.59494 1.78098 L
F
.53165 1.75723 m
.5443 1.75723 L
.5443 1.78098 L
.53165 1.78098 L
F
.63291 1.75723 m
.64557 1.75723 L
.64557 1.78098 L
.63291 1.78098 L
F
.75949 1.73349 m
.77215 1.73349 L
.77215 1.75723 L
.75949 1.75723 L
F
.74684 1.73349 m
.75949 1.73349 L
.75949 1.75723 L
.74684 1.75723 L
F
.58228 1.75723 m
.59494 1.75723 L
.59494 1.78098 L
.58228 1.78098 L
F
.60759 1.75723 m
.62025 1.75723 L
.62025 1.78098 L
.60759 1.78098 L
F
.65823 1.73349 m
.67089 1.73349 L
.67089 1.75723 L
.65823 1.75723 L
F
.63291 1.73349 m
.64557 1.73349 L
.64557 1.75723 L
.63291 1.75723 L
F
.07595 1.75723 m
.08861 1.75723 L
.08861 1.78098 L
.07595 1.78098 L
F
.6962 1.73349 m
.70886 1.73349 L
.70886 1.75723 L
.6962 1.75723 L
F
.64557 1.73349 m
.65823 1.73349 L
.65823 1.75723 L
.64557 1.75723 L
F
.08861 1.75723 m
.10127 1.75723 L
.10127 1.78098 L
.08861 1.78098 L
F
.03797 1.75723 m
.05063 1.75723 L
.05063 1.78098 L
.03797 1.78098 L
F
.06329 1.75723 m
.07595 1.75723 L
.07595 1.78098 L
.06329 1.78098 L
F
.10127 1.75723 m
.11392 1.75723 L
.11392 1.78098 L
.10127 1.78098 L
F
.62025 1.73349 m
.63291 1.73349 L
.63291 1.75723 L
.62025 1.75723 L
F
.59494 1.73349 m
.60759 1.73349 L
.60759 1.75723 L
.59494 1.75723 L
F
.56962 1.73349 m
.58228 1.73349 L
.58228 1.75723 L
.56962 1.75723 L
F
.82278 1.73349 m
.83544 1.73349 L
.83544 1.75723 L
.82278 1.75723 L
F
.60759 1.73349 m
.62025 1.73349 L
.62025 1.75723 L
.60759 1.75723 L
F
.68354 1.73349 m
.6962 1.73349 L
.6962 1.75723 L
.68354 1.75723 L
F
.67089 1.73349 m
.68354 1.73349 L
.68354 1.75723 L
.67089 1.75723 L
F
.55696 1.73349 m
.56962 1.73349 L
.56962 1.75723 L
.55696 1.75723 L
F
.58228 1.73349 m
.59494 1.73349 L
.59494 1.75723 L
.58228 1.75723 L
F
.21519 1.75723 m
.22785 1.75723 L
.22785 1.78098 L
.21519 1.78098 L
F
.18987 1.75723 m
.20253 1.75723 L
.20253 1.78098 L
.18987 1.78098 L
F
.31646 1.75723 m
.32911 1.75723 L
.32911 1.78098 L
.31646 1.78098 L
F
.25316 1.75723 m
.26582 1.75723 L
.26582 1.78098 L
.25316 1.78098 L
F
.20253 1.75723 m
.21519 1.75723 L
.21519 1.78098 L
.20253 1.78098 L
F
.26582 1.75723 m
.27848 1.75723 L
.27848 1.78098 L
.26582 1.78098 L
F
.27848 1.75723 m
.29114 1.75723 L
.29114 1.78098 L
.27848 1.78098 L
F
.29114 1.75723 m
.3038 1.75723 L
.3038 1.78098 L
.29114 1.78098 L
F
.3038 1.75723 m
.31646 1.75723 L
.31646 1.78098 L
.3038 1.78098 L
F
.32911 1.75723 m
.34177 1.75723 L
.34177 1.78098 L
.32911 1.78098 L
F
.49367 1.75723 m
.50633 1.75723 L
.50633 1.78098 L
.49367 1.78098 L
F
.46835 1.75723 m
.48101 1.75723 L
.48101 1.78098 L
.46835 1.78098 L
F
.44304 1.75723 m
.4557 1.75723 L
.4557 1.78098 L
.44304 1.78098 L
F
.36709 1.75723 m
.37975 1.75723 L
.37975 1.78098 L
.36709 1.78098 L
F
.48101 1.75723 m
.49367 1.75723 L
.49367 1.78098 L
.48101 1.78098 L
F
.24051 1.75723 m
.25316 1.75723 L
.25316 1.78098 L
.24051 1.78098 L
F
.22785 1.75723 m
.24051 1.75723 L
.24051 1.78098 L
.22785 1.78098 L
F
.43038 1.75723 m
.44304 1.75723 L
.44304 1.78098 L
.43038 1.78098 L
F
.4557 1.75723 m
.46835 1.75723 L
.46835 1.78098 L
.4557 1.78098 L
F
.77215 1.75723 m
.78481 1.75723 L
.78481 1.78098 L
.77215 1.78098 L
F
.74684 1.75723 m
.75949 1.75723 L
.75949 1.78098 L
.74684 1.78098 L
F
.55696 1.75723 m
.56962 1.75723 L
.56962 1.78098 L
.55696 1.78098 L
F
.81013 1.75723 m
.82278 1.75723 L
.82278 1.78098 L
.81013 1.78098 L
F
.75949 1.75723 m
.77215 1.75723 L
.77215 1.78098 L
.75949 1.78098 L
F
.51899 1.75723 m
.53165 1.75723 L
.53165 1.78098 L
.51899 1.78098 L
F
.50633 1.75723 m
.51899 1.75723 L
.51899 1.78098 L
.50633 1.78098 L
F
.5443 1.75723 m
.55696 1.75723 L
.55696 1.78098 L
.5443 1.78098 L
F
.56962 1.75723 m
.58228 1.75723 L
.58228 1.78098 L
.56962 1.78098 L
F
.72152 1.75723 m
.73418 1.75723 L
.73418 1.78098 L
.72152 1.78098 L
F
.73418 1.75723 m
.74684 1.75723 L
.74684 1.78098 L
.73418 1.78098 L
F
.67089 1.75723 m
.68354 1.75723 L
.68354 1.78098 L
.67089 1.78098 L
F
.70886 1.75723 m
.72152 1.75723 L
.72152 1.78098 L
.70886 1.78098 L
F
.6962 1.75723 m
.70886 1.75723 L
.70886 1.78098 L
.6962 1.78098 L
F
.79747 1.75723 m
.81013 1.75723 L
.81013 1.78098 L
.79747 1.78098 L
F
.78481 1.75723 m
.79747 1.75723 L
.79747 1.78098 L
.78481 1.78098 L
F
.65823 1.75723 m
.67089 1.75723 L
.67089 1.78098 L
.65823 1.78098 L
F
.68354 1.75723 m
.6962 1.75723 L
.6962 1.78098 L
.68354 1.78098 L
F
.02532 1.37729 m
.03797 1.37729 L
.03797 1.40104 L
.02532 1.40104 L
F
.97468 1.35355 m
.98734 1.35355 L
.98734 1.37729 L
.97468 1.37729 L
F
.03797 1.37729 m
.05063 1.37729 L
.05063 1.40104 L
.03797 1.40104 L
F
0 1.37729 m
.01266 1.37729 L
.01266 1.40104 L
0 1.40104 L
F
.96203 1.35355 m
.97468 1.35355 L
.97468 1.37729 L
.96203 1.37729 L
F
.87342 1.35355 m
.88608 1.35355 L
.88608 1.37729 L
.87342 1.37729 L
F
.91139 1.35355 m
.92405 1.35355 L
.92405 1.37729 L
.91139 1.37729 L
F
.88608 1.35355 m
.89873 1.35355 L
.89873 1.37729 L
.88608 1.37729 L
F
.01266 1.37729 m
.02532 1.37729 L
.02532 1.40104 L
.01266 1.40104 L
F
.89873 1.35355 m
.91139 1.35355 L
.91139 1.37729 L
.89873 1.37729 L
F
.6962 1.35355 m
.70886 1.35355 L
.70886 1.37729 L
.6962 1.37729 L
F
.73418 1.35355 m
.74684 1.35355 L
.74684 1.37729 L
.73418 1.37729 L
F
.70886 1.35355 m
.72152 1.35355 L
.72152 1.37729 L
.70886 1.37729 L
F
.83544 1.35355 m
.8481 1.35355 L
.8481 1.37729 L
.83544 1.37729 L
F
.72152 1.35355 m
.73418 1.35355 L
.73418 1.37729 L
.72152 1.37729 L
F
.74684 1.35355 m
.75949 1.35355 L
.75949 1.37729 L
.74684 1.37729 L
F
.98734 1.35355 m
1 1.35355 L
1 1.37729 L
.98734 1.37729 L
F
.75949 1.35355 m
.77215 1.35355 L
.77215 1.37729 L
.75949 1.37729 L
F
.77215 1.35355 m
.78481 1.35355 L
.78481 1.37729 L
.77215 1.37729 L
F
.97468 1.37729 m
.98734 1.37729 L
.98734 1.40104 L
.97468 1.40104 L
F
.01266 1.40104 m
.02532 1.40104 L
.02532 1.42478 L
.01266 1.42478 L
F
.98734 1.37729 m
1 1.37729 L
1 1.40104 L
.98734 1.40104 L
F
.11392 1.40104 m
.12658 1.40104 L
.12658 1.42478 L
.11392 1.42478 L
F
0 1.40104 m
.01266 1.40104 L
.01266 1.42478 L
0 1.42478 L
F
.02532 1.40104 m
.03797 1.40104 L
.03797 1.42478 L
.02532 1.42478 L
F
.26582 1.40104 m
.27848 1.40104 L
.27848 1.42478 L
.26582 1.42478 L
F
.03797 1.40104 m
.05063 1.40104 L
.05063 1.42478 L
.03797 1.42478 L
F
.05063 1.40104 m
.06329 1.40104 L
.06329 1.42478 L
.05063 1.42478 L
F
.08861 1.40104 m
.10127 1.40104 L
.10127 1.42478 L
.08861 1.42478 L
F
.92405 1.35355 m
.93671 1.35355 L
.93671 1.37729 L
.92405 1.37729 L
F
.10127 1.40104 m
.11392 1.40104 L
.11392 1.42478 L
.10127 1.42478 L
F
.93671 1.35355 m
.94937 1.35355 L
.94937 1.37729 L
.93671 1.37729 L
F
.94937 1.35355 m
.96203 1.35355 L
.96203 1.37729 L
.94937 1.37729 L
F
.12658 1.40104 m
.13924 1.40104 L
.13924 1.42478 L
.12658 1.42478 L
F
.13924 1.40104 m
.1519 1.40104 L
.1519 1.42478 L
.13924 1.42478 L
F
.06329 1.40104 m
.07595 1.40104 L
.07595 1.42478 L
.06329 1.42478 L
F
.07595 1.40104 m
.08861 1.40104 L
.08861 1.42478 L
.07595 1.42478 L
F
.18987 1.37729 m
.20253 1.37729 L
.20253 1.40104 L
.18987 1.40104 L
F
.21519 1.37729 m
.22785 1.37729 L
.22785 1.40104 L
.21519 1.40104 L
F
.07595 1.37729 m
.08861 1.37729 L
.08861 1.40104 L
.07595 1.40104 L
F
.1519 1.37729 m
.16456 1.37729 L
.16456 1.40104 L
.1519 1.40104 L
F
.20253 1.37729 m
.21519 1.37729 L
.21519 1.40104 L
.20253 1.40104 L
F
.11392 1.37729 m
.12658 1.37729 L
.12658 1.40104 L
.11392 1.40104 L
F
.06329 1.37729 m
.07595 1.37729 L
.07595 1.40104 L
.06329 1.40104 L
F
.12658 1.37729 m
.13924 1.37729 L
.13924 1.40104 L
.12658 1.40104 L
F
.08861 1.37729 m
.10127 1.37729 L
.10127 1.40104 L
.08861 1.40104 L
F
.05063 1.37729 m
.06329 1.37729 L
.06329 1.40104 L
.05063 1.40104 L
F
.72152 1.23481 m
.73418 1.23481 L
.73418 1.25856 L
.72152 1.25856 L
F
.67089 1.23481 m
.68354 1.23481 L
.68354 1.25856 L
.67089 1.25856 L
F
.73418 1.23481 m
.74684 1.23481 L
.74684 1.25856 L
.73418 1.25856 L
F
.68354 1.23481 m
.6962 1.23481 L
.6962 1.25856 L
.68354 1.25856 L
F
.6962 1.23481 m
.70886 1.23481 L
.70886 1.25856 L
.6962 1.25856 L
F
.17722 1.37729 m
.18987 1.37729 L
.18987 1.40104 L
.17722 1.40104 L
F
.13924 1.37729 m
.1519 1.37729 L
.1519 1.40104 L
.13924 1.40104 L
F
.70886 1.23481 m
.72152 1.23481 L
.72152 1.25856 L
.70886 1.25856 L
F
.16456 1.37729 m
.17722 1.37729 L
.17722 1.40104 L
.16456 1.40104 L
F
.81013 1.35355 m
.82278 1.35355 L
.82278 1.37729 L
.81013 1.37729 L
F
.5443 1.38361 m
.5443 1.40104 L
.53165 1.40104 L
.53165 1.39518 L
F
.82278 1.35355 m
.83544 1.35355 L
.83544 1.37729 L
.82278 1.37729 L
F
.22785 1.37729 m
.24051 1.37729 L
.24051 1.40104 L
.22785 1.40104 L
F
.24051 1.37729 m
.25316 1.37729 L
.25316 1.40104 L
.24051 1.40104 L
F
.8481 1.35355 m
.86076 1.35355 L
.86076 1.37729 L
.8481 1.37729 L
F
.86076 1.35355 m
.87342 1.35355 L
.87342 1.37729 L
.86076 1.37729 L
F
.78481 1.35355 m
.79747 1.35355 L
.79747 1.37729 L
.78481 1.37729 L
F
.79747 1.35355 m
.81013 1.35355 L
.81013 1.37729 L
.79747 1.37729 L
F
.55696 1.37729 m
.56962 1.37729 L
.56962 1.40104 L
.55696 1.40104 L
F
.58228 1.37729 m
.59494 1.37729 L
.59494 1.40104 L
.58228 1.40104 L
F
.60759 1.37729 m
.62025 1.37729 L
.62025 1.40104 L
.60759 1.40104 L
F
.10127 1.37729 m
.11392 1.37729 L
.11392 1.40104 L
.10127 1.40104 L
F
.56962 1.37729 m
.58228 1.37729 L
.58228 1.40104 L
.56962 1.40104 L
F
.25316 1.37729 m
.26582 1.37729 L
.26582 1.40104 L
.25316 1.40104 L
F
.29114 1.39292 m
.29114 1.40104 L
.27848 1.40104 L
.27848 1.38179 L
F
.62025 1.37729 m
.63291 1.37729 L
.63291 1.40104 L
.62025 1.40104 L
F
.59494 1.37729 m
.60759 1.37729 L
.60759 1.40104 L
.59494 1.40104 L
F
.06329 1.3298 m
.07595 1.3298 L
.07595 1.35355 L
.06329 1.35355 L
F
.05063 1.3298 m
.06329 1.3298 L
.06329 1.35355 L
.05063 1.35355 L
F
.11392 1.3298 m
.12658 1.3298 L
.12658 1.35355 L
.11392 1.35355 L
F
.07595 1.3298 m
.08861 1.3298 L
.08861 1.35355 L
.07595 1.35355 L
F
.08861 1.3298 m
.10127 1.3298 L
.10127 1.35355 L
.08861 1.35355 L
F
.67089 1.3298 m
.68354 1.3298 L
.68354 1.35355 L
.67089 1.35355 L
F
.65823 1.3298 m
.67089 1.3298 L
.67089 1.35355 L
.65823 1.35355 L
F
.63291 1.3298 m
.64557 1.3298 L
.64557 1.35355 L
.63291 1.35355 L
F
.12658 1.3298 m
.13924 1.3298 L
.13924 1.35355 L
.12658 1.35355 L
F
.10127 1.3298 m
.11392 1.3298 L
.11392 1.35355 L
.10127 1.35355 L
F
.17722 1.3298 m
.18987 1.3298 L
.18987 1.35355 L
.17722 1.35355 L
F
.16456 1.3298 m
.17722 1.3298 L
.17722 1.35355 L
.16456 1.35355 L
F
.13924 1.3298 m
.1519 1.3298 L
.1519 1.35355 L
.13924 1.35355 L
F
.87342 1.37729 m
.88608 1.37729 L
.88608 1.40104 L
.87342 1.40104 L
F
.8481 1.37729 m
.86076 1.37729 L
.86076 1.40104 L
.8481 1.40104 L
F
.21519 1.3298 m
.22785 1.3298 L
.22785 1.35355 L
.21519 1.35355 L
F
.18987 1.3298 m
.20253 1.3298 L
.20253 1.35355 L
.18987 1.35355 L
F
.1519 1.3298 m
.16456 1.3298 L
.16456 1.35355 L
.1519 1.35355 L
F
.20253 1.3298 m
.21519 1.3298 L
.21519 1.35355 L
.20253 1.35355 L
F
.86076 1.30605 m
.87342 1.30605 L
.87342 1.3298 L
.86076 1.3298 L
F
.81013 1.30605 m
.82278 1.30605 L
.82278 1.3298 L
.81013 1.3298 L
F
.87342 1.30605 m
.88608 1.30605 L
.88608 1.3298 L
.87342 1.3298 L
F
.83544 1.30605 m
.8481 1.30605 L
.8481 1.3298 L
.83544 1.3298 L
F
.79747 1.30605 m
.81013 1.30605 L
.81013 1.3298 L
.79747 1.3298 L
F
.74684 1.30605 m
.75949 1.30605 L
.75949 1.3298 L
.74684 1.3298 L
F
.70886 1.30605 m
.72152 1.30605 L
.72152 1.3298 L
.70886 1.3298 L
F
.8481 1.30605 m
.86076 1.30605 L
.86076 1.3298 L
.8481 1.3298 L
F
.73418 1.30605 m
.74684 1.30605 L
.74684 1.3298 L
.73418 1.3298 L
F
.68354 1.3298 m
.6962 1.3298 L
.6962 1.35355 L
.68354 1.35355 L
F
.70886 1.3298 m
.72152 1.3298 L
.72152 1.35355 L
.70886 1.35355 L
F
.58228 1.3298 m
.59494 1.3298 L
.59494 1.35355 L
.58228 1.35355 L
F
.64557 1.3298 m
.65823 1.3298 L
.65823 1.35355 L
.64557 1.35355 L
F
.6962 1.3298 m
.70886 1.3298 L
.70886 1.35355 L
.6962 1.35355 L
F
.59494 1.3298 m
.60759 1.3298 L
.60759 1.35355 L
.59494 1.35355 L
F
.82278 1.30605 m
.83544 1.30605 L
.83544 1.3298 L
.82278 1.3298 L
F
.60759 1.3298 m
.62025 1.3298 L
.62025 1.35355 L
.60759 1.35355 L
F
.62025 1.3298 m
.63291 1.3298 L
.63291 1.35355 L
.62025 1.35355 L
F
.74684 1.37729 m
.75949 1.37729 L
.75949 1.40104 L
.74684 1.40104 L
F
.20253 1.40104 m
.21519 1.40104 L
.21519 1.42478 L
.20253 1.42478 L
F
.75949 1.37729 m
.77215 1.37729 L
.77215 1.40104 L
.75949 1.40104 L
F
.21519 1.40104 m
.22785 1.40104 L
.22785 1.42478 L
.21519 1.42478 L
F
.22785 1.40104 m
.24051 1.40104 L
.24051 1.42478 L
.22785 1.42478 L
F
.79747 1.37729 m
.81013 1.37729 L
.81013 1.40104 L
.79747 1.40104 L
F
.78481 1.37729 m
.79747 1.37729 L
.79747 1.40104 L
.78481 1.40104 L
F
.77215 1.37729 m
.78481 1.37729 L
.78481 1.40104 L
.77215 1.40104 L
F
.72152 1.37729 m
.73418 1.37729 L
.73418 1.40104 L
.72152 1.40104 L
F
.73418 1.37729 m
.74684 1.37729 L
.74684 1.40104 L
.73418 1.40104 L
F
.32911 1.41887 m
.32911 1.42478 L
.31646 1.42478 L
.31646 1.4114 L
F
.25316 1.40104 m
.26582 1.40104 L
.26582 1.42478 L
.25316 1.42478 L
F
.31646 1.4114 m
.31646 1.42478 L
.3038 1.42478 L
.3038 1.40333 L
F
.27848 1.40104 m
.29114 1.40104 L
.29114 1.42478 L
.27848 1.42478 L
F
.24051 1.40104 m
.25316 1.40104 L
.25316 1.42478 L
.24051 1.42478 L
F
.1519 1.40104 m
.16456 1.40104 L
.16456 1.42478 L
.1519 1.42478 L
F
.16456 1.40104 m
.17722 1.40104 L
.17722 1.42478 L
.16456 1.42478 L
F
.17722 1.40104 m
.18987 1.40104 L
.18987 1.42478 L
.17722 1.42478 L
F
.18987 1.40104 m
.20253 1.40104 L
.20253 1.42478 L
.18987 1.42478 L
F
.96203 1.37729 m
.97468 1.37729 L
.97468 1.40104 L
.96203 1.40104 L
F
.94937 1.37729 m
.96203 1.37729 L
.96203 1.40104 L
.94937 1.40104 L
F
.93671 1.37729 m
.94937 1.37729 L
.94937 1.40104 L
.93671 1.40104 L
F
.88608 1.37729 m
.89873 1.37729 L
.89873 1.40104 L
.88608 1.40104 L
F
.89873 1.37729 m
.91139 1.37729 L
.91139 1.40104 L
.89873 1.40104 L
F
.81013 1.37729 m
.82278 1.37729 L
.82278 1.40104 L
.81013 1.40104 L
F
.86076 1.37729 m
.87342 1.37729 L
.87342 1.40104 L
.86076 1.40104 L
F
.82278 1.37729 m
.83544 1.37729 L
.83544 1.40104 L
.82278 1.40104 L
F
.83544 1.37729 m
.8481 1.37729 L
.8481 1.40104 L
.83544 1.40104 L
F
.64557 1.37729 m
.65823 1.37729 L
.65823 1.40104 L
.64557 1.40104 L
F
.63291 1.37729 m
.64557 1.37729 L
.64557 1.40104 L
.63291 1.40104 L
F
.6962 1.37729 m
.70886 1.37729 L
.70886 1.40104 L
.6962 1.40104 L
F
.65823 1.37729 m
.67089 1.37729 L
.67089 1.40104 L
.65823 1.40104 L
F
.67089 1.37729 m
.68354 1.37729 L
.68354 1.40104 L
.67089 1.40104 L
F
.91139 1.37729 m
.92405 1.37729 L
.92405 1.40104 L
.91139 1.40104 L
F
.92405 1.37729 m
.93671 1.37729 L
.93671 1.40104 L
.92405 1.40104 L
F
.70886 1.37729 m
.72152 1.37729 L
.72152 1.40104 L
.70886 1.40104 L
F
.68354 1.37729 m
.6962 1.37729 L
.6962 1.40104 L
.68354 1.40104 L
F
.34177 1.61476 m
.35443 1.61476 L
.35443 1.6385 L
.34177 1.6385 L
F
.39241 1.61476 m
.40506 1.61476 L
.40506 1.6385 L
.39241 1.6385 L
F
.32911 1.61476 m
.34177 1.61476 L
.34177 1.6385 L
.32911 1.6385 L
F
.36709 1.61476 m
.37975 1.61476 L
.37975 1.6385 L
.36709 1.6385 L
F
.40506 1.61476 m
.41772 1.61476 L
.41772 1.6385 L
.40506 1.6385 L
F
.48101 1.61476 m
.49367 1.61476 L
.49367 1.6385 L
.48101 1.6385 L
F
.4557 1.61476 m
.46835 1.61476 L
.46835 1.6385 L
.4557 1.6385 L
F
.43038 1.61476 m
.44304 1.61476 L
.44304 1.6385 L
.43038 1.6385 L
F
.35443 1.61476 m
.36709 1.61476 L
.36709 1.6385 L
.35443 1.6385 L
F
.46835 1.61476 m
.48101 1.61476 L
.48101 1.6385 L
.46835 1.6385 L
F
.98734 1.59101 m
1 1.59101 L
1 1.61476 L
.98734 1.61476 L
F
.88608 1.59101 m
.89873 1.59101 L
.89873 1.61476 L
.88608 1.61476 L
F
.97468 1.59101 m
.98734 1.59101 L
.98734 1.61476 L
.97468 1.61476 L
F
.87342 1.59101 m
.88608 1.59101 L
.88608 1.61476 L
.87342 1.61476 L
F
.86076 1.59101 m
.87342 1.59101 L
.87342 1.61476 L
.86076 1.61476 L
F
.96203 1.59101 m
.97468 1.59101 L
.97468 1.61476 L
.96203 1.61476 L
F
.93671 1.59101 m
.94937 1.59101 L
.94937 1.61476 L
.93671 1.61476 L
F
.37975 1.61476 m
.39241 1.61476 L
.39241 1.6385 L
.37975 1.6385 L
F
0 1.61476 m
.01266 1.61476 L
.01266 1.6385 L
0 1.6385 L
F
.94937 1.59101 m
.96203 1.59101 L
.96203 1.61476 L
.94937 1.61476 L
F
.25316 1.61476 m
.26582 1.61476 L
.26582 1.6385 L
.25316 1.6385 L
F
.26582 1.61476 m
.27848 1.61476 L
.27848 1.6385 L
.26582 1.6385 L
F
.27848 1.61476 m
.29114 1.61476 L
.29114 1.6385 L
.27848 1.6385 L
F
.29114 1.61476 m
.3038 1.61476 L
.3038 1.6385 L
.29114 1.6385 L
F
.31646 1.61476 m
.32911 1.61476 L
.32911 1.6385 L
.31646 1.6385 L
F
.4557 1.59101 m
.46835 1.59101 L
.46835 1.61476 L
.4557 1.61476 L
F
.46835 1.59101 m
.48101 1.59101 L
.48101 1.61476 L
.46835 1.61476 L
F
.40506 1.59101 m
.41772 1.59101 L
.41772 1.61476 L
.40506 1.61476 L
F
.44304 1.59101 m
.4557 1.59101 L
.4557 1.61476 L
.44304 1.61476 L
F
.43038 1.59101 m
.44304 1.59101 L
.44304 1.61476 L
.43038 1.61476 L
F
.21519 1.61476 m
.22785 1.61476 L
.22785 1.6385 L
.21519 1.6385 L
F
.22785 1.61476 m
.24051 1.61476 L
.24051 1.6385 L
.22785 1.6385 L
F
.24051 1.61476 m
.25316 1.61476 L
.25316 1.6385 L
.24051 1.6385 L
F
.41772 1.61476 m
.43038 1.61476 L
.43038 1.6385 L
.41772 1.6385 L
F
.44304 1.61476 m
.4557 1.61476 L
.4557 1.6385 L
.44304 1.6385 L
F
.20253 1.61476 m
.21519 1.61476 L
.21519 1.6385 L
.20253 1.6385 L
F
.3038 1.61476 m
.31646 1.61476 L
.31646 1.6385 L
.3038 1.6385 L
F
.18987 1.61476 m
.20253 1.61476 L
.20253 1.6385 L
.18987 1.6385 L
F
.17722 1.61476 m
.18987 1.61476 L
.18987 1.6385 L
.17722 1.6385 L
F
.35443 1.56726 m
.36709 1.56726 L
.36709 1.59101 L
.35443 1.59101 L
F
.32911 1.56726 m
.34177 1.56726 L
.34177 1.59101 L
.32911 1.59101 L
F
.3038 1.56726 m
.31646 1.56726 L
.31646 1.59101 L
.3038 1.59101 L
F
.55696 1.56726 m
.56962 1.56726 L
.56962 1.59101 L
.55696 1.59101 L
F
.34177 1.56726 m
.35443 1.56726 L
.35443 1.59101 L
.34177 1.59101 L
F
.40506 1.56726 m
.41772 1.56726 L
.41772 1.59101 L
.40506 1.59101 L
F
.41772 1.56726 m
.43038 1.56726 L
.43038 1.59101 L
.41772 1.59101 L
F
.43038 1.56726 m
.44304 1.56726 L
.44304 1.59101 L
.43038 1.59101 L
F
.29114 1.56726 m
.3038 1.56726 L
.3038 1.59101 L
.29114 1.59101 L
F
.31646 1.56726 m
.32911 1.56726 L
.32911 1.59101 L
.31646 1.59101 L
F
.46835 1.56726 m
.48101 1.56726 L
.48101 1.59101 L
.46835 1.59101 L
F
.44304 1.56726 m
.4557 1.56726 L
.4557 1.59101 L
.44304 1.59101 L
F
.58228 1.56726 m
.59494 1.56726 L
.59494 1.59101 L
.58228 1.59101 L
F
.50633 1.56726 m
.51899 1.56726 L
.51899 1.59101 L
.50633 1.59101 L
F
.4557 1.56726 m
.46835 1.56726 L
.46835 1.59101 L
.4557 1.59101 L
F
.5443 1.56726 m
.55696 1.56726 L
.55696 1.59101 L
.5443 1.59101 L
F
.53165 1.56726 m
.5443 1.56726 L
.5443 1.59101 L
.53165 1.59101 L
F
.56962 1.56726 m
.58228 1.56726 L
.58228 1.59101 L
.56962 1.59101 L
F
.59494 1.56726 m
.60759 1.56726 L
.60759 1.59101 L
.59494 1.59101 L
F
.16456 1.61476 m
.17722 1.61476 L
.17722 1.6385 L
.16456 1.6385 L
F
.13924 1.61476 m
.1519 1.61476 L
.1519 1.6385 L
.13924 1.6385 L
F
.11392 1.61476 m
.12658 1.61476 L
.12658 1.6385 L
.11392 1.6385 L
F
.03797 1.61476 m
.05063 1.61476 L
.05063 1.6385 L
.03797 1.6385 L
F
.1519 1.61476 m
.16456 1.61476 L
.16456 1.6385 L
.1519 1.6385 L
F
.89873 1.59101 m
.91139 1.59101 L
.91139 1.61476 L
.89873 1.61476 L
F
.91139 1.59101 m
.92405 1.59101 L
.92405 1.61476 L
.91139 1.61476 L
F
.92405 1.59101 m
.93671 1.59101 L
.93671 1.61476 L
.92405 1.61476 L
F
.10127 1.61476 m
.11392 1.61476 L
.11392 1.6385 L
.10127 1.6385 L
F
.12658 1.61476 m
.13924 1.61476 L
.13924 1.6385 L
.12658 1.6385 L
F
.06329 1.61476 m
.07595 1.61476 L
.07595 1.6385 L
.06329 1.6385 L
F
.39241 1.56726 m
.40506 1.56726 L
.40506 1.59101 L
.39241 1.59101 L
F
.05063 1.61476 m
.06329 1.61476 L
.06329 1.6385 L
.05063 1.6385 L
F
.37975 1.56726 m
.39241 1.56726 L
.39241 1.59101 L
.37975 1.59101 L
F
.36709 1.56726 m
.37975 1.56726 L
.37975 1.59101 L
.36709 1.59101 L
F
.02532 1.61476 m
.03797 1.61476 L
.03797 1.6385 L
.02532 1.6385 L
F
.01266 1.61476 m
.02532 1.61476 L
.02532 1.6385 L
.01266 1.6385 L
F
.08861 1.61476 m
.10127 1.61476 L
.10127 1.6385 L
.08861 1.6385 L
F
.07595 1.61476 m
.08861 1.61476 L
.08861 1.6385 L
.07595 1.6385 L
F
.8481 .26121 m
.86076 .26121 L
.86076 .28496 L
.8481 .28496 L
F
.86076 .26121 m
.87342 .26121 L
.87342 .28496 L
.86076 .28496 L
F
.87342 .26121 m
.88608 .26121 L
.88608 .28496 L
.87342 .28496 L
F
.74684 .26121 m
.75949 .26121 L
.75949 .28496 L
.74684 .28496 L
F
.77215 .26121 m
.78481 .26121 L
.78481 .28496 L
.77215 .28496 L
F
.64557 .26121 m
.65823 .26121 L
.65823 .28496 L
.64557 .28496 L
F
.65823 .26121 m
.67089 .26121 L
.67089 .28496 L
.65823 .28496 L
F
.67089 .26121 m
.68354 .26121 L
.68354 .28496 L
.67089 .28496 L
F
.82278 .26121 m
.83544 .26121 L
.83544 .28496 L
.82278 .28496 L
F
.83544 .26121 m
.8481 .26121 L
.8481 .28496 L
.83544 .28496 L
F
.64557 1.59101 m
.65823 1.59101 L
.65823 1.61476 L
.64557 1.61476 L
F
.6962 1.59101 m
.70886 1.59101 L
.70886 1.61476 L
.6962 1.61476 L
F
.63291 1.59101 m
.64557 1.59101 L
.64557 1.61476 L
.63291 1.61476 L
F
.68354 1.59101 m
.6962 1.59101 L
.6962 1.61476 L
.68354 1.61476 L
F
.67089 1.59101 m
.68354 1.59101 L
.68354 1.61476 L
.67089 1.61476 L
F
.81013 .26121 m
.82278 .26121 L
.82278 .28496 L
.81013 .28496 L
F
.78481 .26121 m
.79747 .26121 L
.79747 .28496 L
.78481 .28496 L
F
.75949 .26121 m
.77215 .26121 L
.77215 .28496 L
.75949 .28496 L
F
.65823 1.59101 m
.67089 1.59101 L
.67089 1.61476 L
.65823 1.61476 L
F
.79747 .26121 m
.81013 .26121 L
.81013 .28496 L
.79747 .28496 L
F
.02532 .28496 m
.03797 .28496 L
.03797 .3087 L
.02532 .3087 L
F
.03797 .28496 m
.05063 .28496 L
.05063 .3087 L
.03797 .3087 L
F
.05063 .28496 m
.06329 .28496 L
.06329 .3087 L
.05063 .3087 L
F
.06329 .28496 m
.07595 .28496 L
.07595 .3087 L
.06329 .3087 L
F
.08861 .28496 m
.10127 .28496 L
.10127 .3087 L
.08861 .3087 L
F
.10127 .28496 m
.11392 .28496 L
.11392 .3087 L
.10127 .3087 L
F
.1519 .28496 m
.16456 .28496 L
.16456 .3087 L
.1519 .3087 L
F
.11392 .28496 m
.12658 .28496 L
.12658 .3087 L
.11392 .3087 L
F
.13924 .28496 m
.1519 .28496 L
.1519 .3087 L
.13924 .3087 L
F
.12658 .28496 m
.13924 .28496 L
.13924 .3087 L
.12658 .3087 L
F
.72152 .26121 m
.73418 .26121 L
.73418 .28496 L
.72152 .28496 L
F
.63291 .26121 m
.64557 .26121 L
.64557 .28496 L
.63291 .28496 L
F
.70886 .26121 m
.72152 .26121 L
.72152 .28496 L
.70886 .28496 L
F
.62025 .26121 m
.63291 .26121 L
.63291 .28496 L
.62025 .28496 L
F
.60759 .26121 m
.62025 .26121 L
.62025 .28496 L
.60759 .28496 L
F
.6962 .26121 m
.70886 .26121 L
.70886 .28496 L
.6962 .28496 L
F
.07595 .28496 m
.08861 .28496 L
.08861 .3087 L
.07595 .3087 L
F
.73418 .26121 m
.74684 .26121 L
.74684 .28496 L
.73418 .28496 L
F
.68354 .26121 m
.6962 .26121 L
.6962 .28496 L
.68354 .28496 L
F
.24051 1.59101 m
.25316 1.59101 L
.25316 1.61476 L
.24051 1.61476 L
F
.25316 1.59101 m
.26582 1.59101 L
.26582 1.61476 L
.25316 1.61476 L
F
.26582 1.59101 m
.27848 1.59101 L
.27848 1.61476 L
.26582 1.61476 L
F
.27848 1.59101 m
.29114 1.59101 L
.29114 1.61476 L
.27848 1.61476 L
F
.3038 1.59101 m
.31646 1.59101 L
.31646 1.61476 L
.3038 1.61476 L
F
.32911 1.59101 m
.34177 1.59101 L
.34177 1.61476 L
.32911 1.61476 L
F
.37975 1.59101 m
.39241 1.59101 L
.39241 1.61476 L
.37975 1.61476 L
F
.31646 1.59101 m
.32911 1.59101 L
.32911 1.61476 L
.31646 1.61476 L
F
.36709 1.59101 m
.37975 1.59101 L
.37975 1.61476 L
.36709 1.61476 L
F
.35443 1.59101 m
.36709 1.59101 L
.36709 1.61476 L
.35443 1.61476 L
F
.51899 1.59101 m
.53165 1.59101 L
.53165 1.61476 L
.51899 1.61476 L
F
.53165 1.59101 m
.5443 1.59101 L
.5443 1.61476 L
.53165 1.61476 L
F
.5443 1.59101 m
.55696 1.59101 L
.55696 1.61476 L
.5443 1.61476 L
F
.39241 1.59101 m
.40506 1.59101 L
.40506 1.61476 L
.39241 1.61476 L
F
.41772 1.59101 m
.43038 1.59101 L
.43038 1.61476 L
.41772 1.61476 L
F
.50633 1.59101 m
.51899 1.59101 L
.51899 1.61476 L
.50633 1.61476 L
F
.29114 1.59101 m
.3038 1.59101 L
.3038 1.61476 L
.29114 1.61476 L
F
.49367 1.59101 m
.50633 1.59101 L
.50633 1.61476 L
.49367 1.61476 L
F
.48101 1.59101 m
.49367 1.59101 L
.49367 1.61476 L
.48101 1.61476 L
F
.81013 1.59101 m
.82278 1.59101 L
.82278 1.61476 L
.81013 1.61476 L
F
.78481 1.59101 m
.79747 1.59101 L
.79747 1.61476 L
.78481 1.61476 L
F
.60759 1.59101 m
.62025 1.59101 L
.62025 1.61476 L
.60759 1.61476 L
F
.8481 1.59101 m
.86076 1.59101 L
.86076 1.61476 L
.8481 1.61476 L
F
.79747 1.59101 m
.81013 1.59101 L
.81013 1.61476 L
.79747 1.61476 L
F
.55696 1.59101 m
.56962 1.59101 L
.56962 1.61476 L
.55696 1.61476 L
F
.56962 1.59101 m
.58228 1.59101 L
.58228 1.61476 L
.56962 1.61476 L
F
.58228 1.59101 m
.59494 1.59101 L
.59494 1.61476 L
.58228 1.61476 L
F
.59494 1.59101 m
.60759 1.59101 L
.60759 1.61476 L
.59494 1.61476 L
F
.62025 1.59101 m
.63291 1.59101 L
.63291 1.61476 L
.62025 1.61476 L
F
.77215 1.59101 m
.78481 1.59101 L
.78481 1.61476 L
.77215 1.61476 L
F
.74684 1.59101 m
.75949 1.59101 L
.75949 1.61476 L
.74684 1.61476 L
F
.72152 1.59101 m
.73418 1.59101 L
.73418 1.61476 L
.72152 1.61476 L
F
.34177 1.59101 m
.35443 1.59101 L
.35443 1.61476 L
.34177 1.61476 L
F
.75949 1.59101 m
.77215 1.59101 L
.77215 1.61476 L
.75949 1.61476 L
F
.83544 1.59101 m
.8481 1.59101 L
.8481 1.61476 L
.83544 1.61476 L
F
.82278 1.59101 m
.83544 1.59101 L
.83544 1.61476 L
.82278 1.61476 L
F
.70886 1.59101 m
.72152 1.59101 L
.72152 1.61476 L
.70886 1.61476 L
F
.73418 1.59101 m
.74684 1.59101 L
.74684 1.61476 L
.73418 1.61476 L
F
.88608 1.6385 m
.89873 1.6385 L
.89873 1.66225 L
.88608 1.66225 L
F
.78481 1.6385 m
.79747 1.6385 L
.79747 1.66225 L
.78481 1.66225 L
F
.87342 1.6385 m
.88608 1.6385 L
.88608 1.66225 L
.87342 1.66225 L
F
.77215 1.6385 m
.78481 1.6385 L
.78481 1.66225 L
.77215 1.66225 L
F
.75949 1.6385 m
.77215 1.6385 L
.77215 1.66225 L
.75949 1.66225 L
F
.86076 1.6385 m
.87342 1.6385 L
.87342 1.66225 L
.86076 1.66225 L
F
.83544 1.6385 m
.8481 1.6385 L
.8481 1.66225 L
.83544 1.66225 L
F
.27848 1.66225 m
.29114 1.66225 L
.29114 1.686 L
.27848 1.686 L
F
.89873 1.6385 m
.91139 1.6385 L
.91139 1.66225 L
.89873 1.66225 L
F
.8481 1.6385 m
.86076 1.6385 L
.86076 1.66225 L
.8481 1.66225 L
F
.06329 1.66225 m
.07595 1.66225 L
.07595 1.686 L
.06329 1.686 L
F
.03797 1.66225 m
.05063 1.66225 L
.05063 1.686 L
.03797 1.686 L
F
.01266 1.66225 m
.02532 1.66225 L
.02532 1.686 L
.01266 1.686 L
F
.93671 1.6385 m
.94937 1.6385 L
.94937 1.66225 L
.93671 1.66225 L
F
.05063 1.66225 m
.06329 1.66225 L
.06329 1.686 L
.05063 1.686 L
F
.79747 1.6385 m
.81013 1.6385 L
.81013 1.66225 L
.79747 1.66225 L
F
.81013 1.6385 m
.82278 1.6385 L
.82278 1.66225 L
.81013 1.66225 L
F
.82278 1.6385 m
.83544 1.6385 L
.83544 1.66225 L
.82278 1.66225 L
F
0 1.66225 m
.01266 1.66225 L
.01266 1.686 L
0 1.686 L
F
.02532 1.66225 m
.03797 1.66225 L
.03797 1.686 L
.02532 1.686 L
F
.12658 1.66225 m
.13924 1.66225 L
.13924 1.686 L
.12658 1.686 L
F
.34177 1.66225 m
.35443 1.66225 L
.35443 1.686 L
.34177 1.686 L
F
.11392 1.66225 m
.12658 1.66225 L
.12658 1.686 L
.11392 1.686 L
F
.32911 1.66225 m
.34177 1.66225 L
.34177 1.686 L
.32911 1.686 L
F
.31646 1.66225 m
.32911 1.66225 L
.32911 1.686 L
.31646 1.686 L
F
.10127 1.66225 m
.11392 1.66225 L
.11392 1.686 L
.10127 1.686 L
F
.07595 1.66225 m
.08861 1.66225 L
.08861 1.686 L
.07595 1.686 L
F
.20253 1.66225 m
.21519 1.66225 L
.21519 1.686 L
.20253 1.686 L
F
.13924 1.66225 m
.1519 1.66225 L
.1519 1.686 L
.13924 1.686 L
F
.08861 1.66225 m
.10127 1.66225 L
.10127 1.686 L
.08861 1.686 L
F
.24051 1.66225 m
.25316 1.66225 L
.25316 1.686 L
.24051 1.686 L
F
.29114 1.66225 m
.3038 1.66225 L
.3038 1.686 L
.29114 1.686 L
F
.22785 1.66225 m
.24051 1.66225 L
.24051 1.686 L
.22785 1.686 L
F
.26582 1.66225 m
.27848 1.66225 L
.27848 1.686 L
.26582 1.686 L
F
.3038 1.66225 m
.31646 1.66225 L
.31646 1.686 L
.3038 1.686 L
F
.35443 1.66225 m
.36709 1.66225 L
.36709 1.686 L
.35443 1.686 L
F
.37975 1.66225 m
.39241 1.66225 L
.39241 1.686 L
.37975 1.686 L
F
.25316 1.66225 m
.26582 1.66225 L
.26582 1.686 L
.25316 1.686 L
F
.36709 1.66225 m
.37975 1.66225 L
.37975 1.686 L
.36709 1.686 L
F
.86076 1.66225 m
.87342 1.66225 L
.87342 1.686 L
.86076 1.686 L
F
.87342 1.66225 m
.88608 1.66225 L
.88608 1.686 L
.87342 1.686 L
F
.88608 1.66225 m
.89873 1.66225 L
.89873 1.686 L
.88608 1.686 L
F
.93671 1.66225 m
.94937 1.66225 L
.94937 1.686 L
.93671 1.686 L
F
.92405 1.66225 m
.93671 1.66225 L
.93671 1.686 L
.92405 1.686 L
F
.96203 1.66225 m
.97468 1.66225 L
.97468 1.686 L
.96203 1.686 L
F
.01266 1.686 m
.02532 1.686 L
.02532 1.70974 L
.01266 1.70974 L
F
.94937 1.66225 m
.96203 1.66225 L
.96203 1.686 L
.94937 1.686 L
F
0 1.686 m
.01266 1.686 L
.01266 1.70974 L
0 1.70974 L
F
.98734 1.66225 m
1 1.66225 L
1 1.686 L
.98734 1.686 L
F
.53165 1.66225 m
.5443 1.66225 L
.5443 1.686 L
.53165 1.686 L
F
.50633 1.66225 m
.51899 1.66225 L
.51899 1.686 L
.50633 1.686 L
F
.48101 1.66225 m
.49367 1.66225 L
.49367 1.686 L
.48101 1.686 L
F
.41772 1.66225 m
.43038 1.66225 L
.43038 1.686 L
.41772 1.686 L
F
.51899 1.66225 m
.53165 1.66225 L
.53165 1.686 L
.51899 1.686 L
F
.91139 1.66225 m
.92405 1.66225 L
.92405 1.686 L
.91139 1.686 L
F
.89873 1.66225 m
.91139 1.66225 L
.91139 1.686 L
.89873 1.686 L
F
.46835 1.66225 m
.48101 1.66225 L
.48101 1.686 L
.46835 1.686 L
F
.49367 1.66225 m
.50633 1.66225 L
.50633 1.686 L
.49367 1.686 L
F
.81013 1.66225 m
.82278 1.66225 L
.82278 1.686 L
.81013 1.686 L
F
.78481 1.66225 m
.79747 1.66225 L
.79747 1.686 L
.78481 1.686 L
F
.96203 1.6385 m
.97468 1.6385 L
.97468 1.66225 L
.96203 1.66225 L
F
.8481 1.66225 m
.86076 1.66225 L
.86076 1.686 L
.8481 1.686 L
F
.79747 1.66225 m
.81013 1.66225 L
.81013 1.686 L
.79747 1.686 L
F
.92405 1.6385 m
.93671 1.6385 L
.93671 1.66225 L
.92405 1.66225 L
F
.97468 1.6385 m
.98734 1.6385 L
.98734 1.66225 L
.97468 1.66225 L
F
.91139 1.6385 m
.92405 1.6385 L
.92405 1.66225 L
.91139 1.66225 L
F
.94937 1.6385 m
.96203 1.6385 L
.96203 1.66225 L
.94937 1.66225 L
F
.98734 1.6385 m
1 1.6385 L
1 1.66225 L
.98734 1.66225 L
F
.77215 1.66225 m
.78481 1.66225 L
.78481 1.686 L
.77215 1.686 L
F
.74684 1.66225 m
.75949 1.66225 L
.75949 1.686 L
.74684 1.686 L
F
.72152 1.66225 m
.73418 1.66225 L
.73418 1.686 L
.72152 1.686 L
F
.97468 1.66225 m
.98734 1.66225 L
.98734 1.686 L
.97468 1.686 L
F
.75949 1.66225 m
.77215 1.66225 L
.77215 1.686 L
.75949 1.686 L
F
.83544 1.66225 m
.8481 1.66225 L
.8481 1.686 L
.83544 1.686 L
F
.82278 1.66225 m
.83544 1.66225 L
.83544 1.686 L
.82278 1.686 L
F
.70886 1.66225 m
.72152 1.66225 L
.72152 1.686 L
.70886 1.686 L
F
.73418 1.66225 m
.74684 1.66225 L
.74684 1.686 L
.73418 1.686 L
F
.20253 1.56726 m
.21519 1.56726 L
.21519 1.59101 L
.20253 1.59101 L
F
.16456 1.56726 m
.17722 1.56726 L
.17722 1.59101 L
.16456 1.59101 L
F
.18987 1.56726 m
.20253 1.56726 L
.20253 1.59101 L
.18987 1.59101 L
F
.02532 1.59101 m
.03797 1.59101 L
.03797 1.61476 L
.02532 1.61476 L
F
.17722 1.56726 m
.18987 1.56726 L
.18987 1.59101 L
.17722 1.59101 L
F
.26582 1.56726 m
.27848 1.56726 L
.27848 1.59101 L
.26582 1.59101 L
F
.1519 1.56726 m
.16456 1.56726 L
.16456 1.59101 L
.1519 1.59101 L
F
.25316 1.56726 m
.26582 1.56726 L
.26582 1.59101 L
.25316 1.59101 L
F
.13924 1.56726 m
.1519 1.56726 L
.1519 1.59101 L
.13924 1.59101 L
F
.12658 1.56726 m
.13924 1.56726 L
.13924 1.59101 L
.12658 1.59101 L
F
.94937 1.56726 m
.96203 1.56726 L
.96203 1.59101 L
.94937 1.59101 L
F
.92405 1.56726 m
.93671 1.56726 L
.93671 1.59101 L
.92405 1.59101 L
F
.05063 1.59101 m
.06329 1.59101 L
.06329 1.61476 L
.05063 1.61476 L
F
.98734 1.56726 m
1 1.56726 L
1 1.59101 L
.98734 1.59101 L
F
.93671 1.56726 m
.94937 1.56726 L
.94937 1.59101 L
.93671 1.59101 L
F
.01266 1.59101 m
.02532 1.59101 L
.02532 1.61476 L
.01266 1.61476 L
F
0 1.59101 m
.01266 1.59101 L
.01266 1.61476 L
0 1.61476 L
F
.03797 1.59101 m
.05063 1.59101 L
.05063 1.61476 L
.03797 1.61476 L
F
.06329 1.59101 m
.07595 1.59101 L
.07595 1.61476 L
.06329 1.61476 L
F
.11392 1.56726 m
.12658 1.56726 L
.12658 1.59101 L
.11392 1.59101 L
F
.08861 1.56726 m
.10127 1.56726 L
.10127 1.59101 L
.08861 1.59101 L
F
.06329 1.56726 m
.07595 1.56726 L
.07595 1.59101 L
.06329 1.59101 L
F
0 1.56726 m
.01266 1.56726 L
.01266 1.59101 L
0 1.59101 L
F
.10127 1.56726 m
.11392 1.56726 L
.11392 1.59101 L
.10127 1.59101 L
F
.48101 1.56726 m
.49367 1.56726 L
.49367 1.59101 L
.48101 1.59101 L
F
.49367 1.56726 m
.50633 1.56726 L
.50633 1.59101 L
.49367 1.59101 L
F
.51899 1.56726 m
.53165 1.56726 L
.53165 1.59101 L
.51899 1.59101 L
F
.05063 1.56726 m
.06329 1.56726 L
.06329 1.59101 L
.05063 1.59101 L
F
.07595 1.56726 m
.08861 1.56726 L
.08861 1.59101 L
.07595 1.59101 L
F
.24051 1.56726 m
.25316 1.56726 L
.25316 1.59101 L
.24051 1.59101 L
F
.21519 1.56726 m
.22785 1.56726 L
.22785 1.59101 L
.21519 1.59101 L
F
.02532 1.56726 m
.03797 1.56726 L
.03797 1.59101 L
.02532 1.59101 L
F
.27848 1.56726 m
.29114 1.56726 L
.29114 1.59101 L
.27848 1.59101 L
F
.22785 1.56726 m
.24051 1.56726 L
.24051 1.59101 L
.22785 1.59101 L
F
.98734 1.54352 m
1 1.54352 L
1 1.56726 L
.98734 1.56726 L
F
.97468 1.54352 m
.98734 1.54352 L
.98734 1.56726 L
.97468 1.56726 L
F
.01266 1.56726 m
.02532 1.56726 L
.02532 1.59101 L
.01266 1.59101 L
F
.03797 1.56726 m
.05063 1.56726 L
.05063 1.59101 L
.03797 1.59101 L
F
.86076 1.56726 m
.87342 1.56726 L
.87342 1.59101 L
.86076 1.59101 L
F
.91139 1.56726 m
.92405 1.56726 L
.92405 1.59101 L
.91139 1.59101 L
F
.8481 1.56726 m
.86076 1.56726 L
.86076 1.59101 L
.8481 1.59101 L
F
.89873 1.56726 m
.91139 1.56726 L
.91139 1.59101 L
.89873 1.59101 L
F
.88608 1.56726 m
.89873 1.56726 L
.89873 1.59101 L
.88608 1.59101 L
F
.67089 1.56726 m
.68354 1.56726 L
.68354 1.59101 L
.67089 1.59101 L
F
.64557 1.56726 m
.65823 1.56726 L
.65823 1.59101 L
.64557 1.59101 L
F
.62025 1.56726 m
.63291 1.56726 L
.63291 1.59101 L
.62025 1.59101 L
F
.87342 1.56726 m
.88608 1.56726 L
.88608 1.59101 L
.87342 1.59101 L
F
.65823 1.56726 m
.67089 1.56726 L
.67089 1.59101 L
.65823 1.59101 L
F
.1519 1.66225 m
.16456 1.66225 L
.16456 1.686 L
.1519 1.686 L
F
.16456 1.66225 m
.17722 1.66225 L
.17722 1.686 L
.16456 1.686 L
F
.17722 1.66225 m
.18987 1.66225 L
.18987 1.686 L
.17722 1.686 L
F
.18987 1.66225 m
.20253 1.66225 L
.20253 1.686 L
.18987 1.686 L
F
.21519 1.66225 m
.22785 1.66225 L
.22785 1.686 L
.21519 1.686 L
F
.83544 1.56726 m
.8481 1.56726 L
.8481 1.59101 L
.83544 1.59101 L
F
.82278 1.56726 m
.83544 1.56726 L
.83544 1.59101 L
.82278 1.59101 L
F
.81013 1.56726 m
.82278 1.56726 L
.82278 1.59101 L
.81013 1.59101 L
F
.79747 1.56726 m
.81013 1.56726 L
.81013 1.59101 L
.79747 1.59101 L
F
.20253 1.59101 m
.21519 1.59101 L
.21519 1.61476 L
.20253 1.61476 L
F
.21519 1.59101 m
.22785 1.59101 L
.22785 1.61476 L
.21519 1.61476 L
F
.22785 1.59101 m
.24051 1.59101 L
.24051 1.61476 L
.22785 1.61476 L
F
.07595 1.59101 m
.08861 1.59101 L
.08861 1.61476 L
.07595 1.61476 L
F
.10127 1.59101 m
.11392 1.59101 L
.11392 1.61476 L
.10127 1.61476 L
F
.97468 1.56726 m
.98734 1.56726 L
.98734 1.59101 L
.97468 1.59101 L
F
.18987 1.59101 m
.20253 1.59101 L
.20253 1.61476 L
.18987 1.61476 L
F
.96203 1.56726 m
.97468 1.56726 L
.97468 1.59101 L
.96203 1.59101 L
F
.17722 1.59101 m
.18987 1.59101 L
.18987 1.61476 L
.17722 1.61476 L
F
.16456 1.59101 m
.17722 1.59101 L
.17722 1.61476 L
.16456 1.61476 L
F
.68354 1.56726 m
.6962 1.56726 L
.6962 1.59101 L
.68354 1.59101 L
F
.6962 1.56726 m
.70886 1.56726 L
.70886 1.59101 L
.6962 1.59101 L
F
.12658 1.59101 m
.13924 1.59101 L
.13924 1.61476 L
.12658 1.61476 L
F
.60759 1.56726 m
.62025 1.56726 L
.62025 1.59101 L
.60759 1.59101 L
F
.63291 1.56726 m
.64557 1.56726 L
.64557 1.59101 L
.63291 1.59101 L
F
.13924 1.59101 m
.1519 1.59101 L
.1519 1.61476 L
.13924 1.61476 L
F
.08861 1.59101 m
.10127 1.59101 L
.10127 1.61476 L
.08861 1.61476 L
F
.11392 1.59101 m
.12658 1.59101 L
.12658 1.61476 L
.11392 1.61476 L
F
.1519 1.59101 m
.16456 1.59101 L
.16456 1.61476 L
.1519 1.61476 L
F
.26582 .21372 m
.27848 .21372 L
.27848 .23746 L
.26582 .23746 L
F
.31646 .21372 m
.32911 .21372 L
.32911 .23746 L
.31646 .23746 L
F
.25316 .21372 m
.26582 .21372 L
.26582 .23746 L
.25316 .23746 L
F
.3038 .21372 m
.31646 .21372 L
.31646 .23746 L
.3038 .23746 L
F
.29114 .21372 m
.3038 .21372 L
.3038 .23746 L
.29114 .23746 L
F
.37975 .21372 m
.39241 .21372 L
.39241 .23746 L
.37975 .23746 L
F
.35443 .21372 m
.36709 .21372 L
.36709 .23746 L
.35443 .23746 L
F
.32911 .21372 m
.34177 .21372 L
.34177 .23746 L
.32911 .23746 L
F
.27848 .21372 m
.29114 .21372 L
.29114 .23746 L
.27848 .23746 L
F
.36709 .21372 m
.37975 .21372 L
.37975 .23746 L
.36709 .23746 L
F
.83544 .18997 m
.8481 .18997 L
.8481 .21372 L
.83544 .21372 L
F
.8481 .18997 m
.86076 .18997 L
.86076 .21372 L
.8481 .21372 L
F
.86076 .18997 m
.87342 .18997 L
.87342 .21372 L
.86076 .21372 L
F
.87342 .18997 m
.88608 .18997 L
.88608 .21372 L
.87342 .21372 L
F
.89873 .18997 m
.91139 .18997 L
.91139 .21372 L
.89873 .21372 L
F
.91139 .18997 m
.92405 .18997 L
.92405 .21372 L
.91139 .21372 L
F
.96203 .18997 m
.97468 .18997 L
.97468 .21372 L
.96203 .21372 L
F
.92405 .18997 m
.93671 .18997 L
.93671 .21372 L
.92405 .21372 L
F
.94937 .18997 m
.96203 .18997 L
.96203 .21372 L
.94937 .21372 L
F
.93671 .18997 m
.94937 .18997 L
.94937 .21372 L
.93671 .21372 L
F
.68354 .23746 m
.6962 .23746 L
.6962 .26121 L
.68354 .26121 L
F
.20253 .21372 m
.21519 .21372 L
.21519 .23746 L
.20253 .23746 L
F
.67089 .23746 m
.68354 .23746 L
.68354 .26121 L
.67089 .26121 L
F
.24051 .21372 m
.25316 .21372 L
.25316 .23746 L
.24051 .23746 L
F
.18987 .21372 m
.20253 .21372 L
.20253 .23746 L
.18987 .23746 L
F
.65823 .23746 m
.67089 .23746 L
.67089 .26121 L
.65823 .26121 L
F
.63291 .23746 m
.64557 .23746 L
.64557 .26121 L
.63291 .26121 L
F
.74684 .23746 m
.75949 .23746 L
.75949 .26121 L
.74684 .26121 L
F
.6962 .23746 m
.70886 .23746 L
.70886 .26121 L
.6962 .26121 L
F
.64557 .23746 m
.65823 .23746 L
.65823 .26121 L
.64557 .26121 L
F
.17722 .21372 m
.18987 .21372 L
.18987 .23746 L
.17722 .23746 L
F
.1519 .21372 m
.16456 .21372 L
.16456 .23746 L
.1519 .23746 L
F
.12658 .21372 m
.13924 .21372 L
.13924 .23746 L
.12658 .23746 L
F
.34177 .21372 m
.35443 .21372 L
.35443 .23746 L
.34177 .23746 L
F
.16456 .21372 m
.17722 .21372 L
.17722 .23746 L
.16456 .23746 L
F
.22785 .21372 m
.24051 .21372 L
.24051 .23746 L
.22785 .23746 L
F
.21519 .21372 m
.22785 .21372 L
.22785 .23746 L
.21519 .23746 L
F
.11392 .21372 m
.12658 .21372 L
.12658 .23746 L
.11392 .23746 L
F
.13924 .21372 m
.1519 .21372 L
.1519 .23746 L
.13924 .23746 L
F
.93671 .21372 m
.94937 .21372 L
.94937 .23746 L
.93671 .23746 L
F
.91139 .21372 m
.92405 .21372 L
.92405 .23746 L
.91139 .23746 L
F
.88608 .21372 m
.89873 .21372 L
.89873 .23746 L
.88608 .23746 L
F
.83544 .21372 m
.8481 .21372 L
.8481 .23746 L
.83544 .23746 L
F
.92405 .21372 m
.93671 .21372 L
.93671 .23746 L
.92405 .23746 L
F
.73418 .21372 m
.74684 .21372 L
.74684 .23746 L
.73418 .23746 L
F
.70886 .21372 m
.72152 .21372 L
.72152 .23746 L
.70886 .23746 L
F
.68354 .21372 m
.6962 .21372 L
.6962 .23746 L
.68354 .23746 L
F
.89873 .21372 m
.91139 .21372 L
.91139 .23746 L
.89873 .23746 L
F
.72152 .21372 m
.73418 .21372 L
.73418 .23746 L
.72152 .23746 L
F
.48101 .21372 m
.49367 .21372 L
.49367 .23746 L
.48101 .23746 L
F
.46835 .21372 m
.48101 .21372 L
.48101 .23746 L
.46835 .23746 L
F
.86076 .21372 m
.87342 .21372 L
.87342 .23746 L
.86076 .23746 L
F
.49367 .21372 m
.50633 .21372 L
.50633 .23746 L
.49367 .23746 L
F
.51899 .21372 m
.53165 .21372 L
.53165 .23746 L
.51899 .23746 L
F
.82278 .21372 m
.83544 .21372 L
.83544 .23746 L
.82278 .23746 L
F
.81013 .21372 m
.82278 .21372 L
.82278 .23746 L
.81013 .23746 L
F
.8481 .21372 m
.86076 .21372 L
.86076 .23746 L
.8481 .23746 L
F
.87342 .21372 m
.88608 .21372 L
.88608 .23746 L
.87342 .23746 L
F
0 .21372 m
.01266 .21372 L
.01266 .23746 L
0 .23746 L
F
.97468 .18997 m
.98734 .18997 L
.98734 .21372 L
.97468 .21372 L
F
.08861 .21372 m
.10127 .21372 L
.10127 .23746 L
.08861 .23746 L
F
.03797 .21372 m
.05063 .21372 L
.05063 .23746 L
.03797 .23746 L
F
.98734 .18997 m
1 .18997 L
1 .21372 L
.98734 .21372 L
F
.06329 .21372 m
.07595 .21372 L
.07595 .23746 L
.06329 .23746 L
F
.05063 .21372 m
.06329 .21372 L
.06329 .23746 L
.05063 .23746 L
F
.88608 .18997 m
.89873 .18997 L
.89873 .21372 L
.88608 .21372 L
F
.07595 .21372 m
.08861 .21372 L
.08861 .23746 L
.07595 .23746 L
F
.10127 .21372 m
.11392 .21372 L
.11392 .23746 L
.10127 .23746 L
F
.77215 .21372 m
.78481 .21372 L
.78481 .23746 L
.77215 .23746 L
F
.78481 .21372 m
.79747 .21372 L
.79747 .23746 L
.78481 .23746 L
F
.79747 .21372 m
.81013 .21372 L
.81013 .23746 L
.79747 .23746 L
F
.67089 .21372 m
.68354 .21372 L
.68354 .23746 L
.67089 .23746 L
F
.6962 .21372 m
.70886 .21372 L
.70886 .23746 L
.6962 .23746 L
F
.02532 .21372 m
.03797 .21372 L
.03797 .23746 L
.02532 .23746 L
F
.01266 .21372 m
.02532 .21372 L
.02532 .23746 L
.01266 .23746 L
F
.74684 .21372 m
.75949 .21372 L
.75949 .23746 L
.74684 .23746 L
F
.75949 .21372 m
.77215 .21372 L
.77215 .23746 L
.75949 .23746 L
F
.01266 .23746 m
.02532 .23746 L
.02532 .26121 L
.01266 .26121 L
F
.98734 .21372 m
1 .21372 L
1 .23746 L
.98734 .23746 L
F
.96203 .21372 m
.97468 .21372 L
.97468 .23746 L
.96203 .23746 L
F
.17722 .23746 m
.18987 .23746 L
.18987 .26121 L
.17722 .26121 L
F
0 .23746 m
.01266 .23746 L
.01266 .26121 L
0 .26121 L
F
.05063 .23746 m
.06329 .23746 L
.06329 .26121 L
.05063 .26121 L
F
.06329 .23746 m
.07595 .23746 L
.07595 .26121 L
.06329 .26121 L
F
.07595 .23746 m
.08861 .23746 L
.08861 .26121 L
.07595 .26121 L
F
.94937 .21372 m
.96203 .21372 L
.96203 .23746 L
.94937 .23746 L
F
.97468 .21372 m
.98734 .21372 L
.98734 .23746 L
.97468 .23746 L
F
.08861 .23746 m
.10127 .23746 L
.10127 .26121 L
.08861 .26121 L
F
.10127 .23746 m
.11392 .23746 L
.11392 .26121 L
.10127 .26121 L
F
.11392 .23746 m
.12658 .23746 L
.12658 .26121 L
.11392 .26121 L
F
.12658 .23746 m
.13924 .23746 L
.13924 .26121 L
.12658 .26121 L
F
.1519 .23746 m
.16456 .23746 L
.16456 .26121 L
.1519 .26121 L
F
.21519 .23746 m
.22785 .23746 L
.22785 .26121 L
.21519 .26121 L
F
.16456 .23746 m
.17722 .23746 L
.17722 .26121 L
.16456 .26121 L
F
.20253 .23746 m
.21519 .23746 L
.21519 .26121 L
.20253 .26121 L
F
.18987 .23746 m
.20253 .23746 L
.20253 .26121 L
.18987 .26121 L
F
.44304 .23746 m
.4557 .23746 L
.4557 .26121 L
.44304 .26121 L
F
.43038 .23746 m
.44304 .23746 L
.44304 .26121 L
.43038 .26121 L
F
.26582 .23746 m
.27848 .23746 L
.27848 .26121 L
.26582 .26121 L
F
.4557 .23746 m
.46835 .23746 L
.46835 .26121 L
.4557 .26121 L
F
.48101 .23746 m
.49367 .23746 L
.49367 .26121 L
.48101 .26121 L
F
.24051 .23746 m
.25316 .23746 L
.25316 .26121 L
.24051 .26121 L
F
.22785 .23746 m
.24051 .23746 L
.24051 .26121 L
.22785 .26121 L
F
.32911 .23746 m
.34177 .23746 L
.34177 .26121 L
.32911 .26121 L
F
.25316 .23746 m
.26582 .23746 L
.26582 .26121 L
.25316 .26121 L
F
.27848 .23746 m
.29114 .23746 L
.29114 .26121 L
.27848 .26121 L
F
.39241 .23746 m
.40506 .23746 L
.40506 .26121 L
.39241 .26121 L
F
.40506 .23746 m
.41772 .23746 L
.41772 .26121 L
.40506 .26121 L
F
.41772 .23746 m
.43038 .23746 L
.43038 .26121 L
.41772 .26121 L
F
.02532 .23746 m
.03797 .23746 L
.03797 .26121 L
.02532 .26121 L
F
.03797 .23746 m
.05063 .23746 L
.05063 .26121 L
.03797 .26121 L
F
.37975 .23746 m
.39241 .23746 L
.39241 .26121 L
.37975 .26121 L
F
.46835 .23746 m
.48101 .23746 L
.48101 .26121 L
.46835 .26121 L
F
.36709 .23746 m
.37975 .23746 L
.37975 .26121 L
.36709 .26121 L
F
.35443 .23746 m
.36709 .23746 L
.36709 .26121 L
.35443 .26121 L
F
.62025 .23746 m
.63291 .23746 L
.63291 .26121 L
.62025 .26121 L
F
.59494 .23746 m
.60759 .23746 L
.60759 .26121 L
.59494 .26121 L
F
.56962 .23746 m
.58228 .23746 L
.58228 .26121 L
.56962 .26121 L
F
.51899 .23746 m
.53165 .23746 L
.53165 .26121 L
.51899 .26121 L
F
.60759 .23746 m
.62025 .23746 L
.62025 .26121 L
.60759 .26121 L
F
.97468 .23746 m
.98734 .23746 L
.98734 .26121 L
.97468 .26121 L
F
.94937 .23746 m
.96203 .23746 L
.96203 .26121 L
.94937 .26121 L
F
.92405 .23746 m
.93671 .23746 L
.93671 .26121 L
.92405 .26121 L
F
.58228 .23746 m
.59494 .23746 L
.59494 .26121 L
.58228 .26121 L
F
.96203 .23746 m
.97468 .23746 L
.97468 .26121 L
.96203 .26121 L
F
.72152 .23746 m
.73418 .23746 L
.73418 .26121 L
.72152 .26121 L
F
.70886 .23746 m
.72152 .23746 L
.72152 .26121 L
.70886 .26121 L
F
.5443 .23746 m
.55696 .23746 L
.55696 .26121 L
.5443 .26121 L
F
.73418 .23746 m
.74684 .23746 L
.74684 .26121 L
.73418 .26121 L
F
.75949 .23746 m
.77215 .23746 L
.77215 .26121 L
.75949 .26121 L
F
.50633 .23746 m
.51899 .23746 L
.51899 .26121 L
.50633 .26121 L
F
.49367 .23746 m
.50633 .23746 L
.50633 .26121 L
.49367 .26121 L
F
.53165 .23746 m
.5443 .23746 L
.5443 .26121 L
.53165 .26121 L
F
.55696 .23746 m
.56962 .23746 L
.56962 .26121 L
.55696 .26121 L
F
.79747 .23746 m
.81013 .23746 L
.81013 .26121 L
.79747 .26121 L
F
.77215 .23746 m
.78481 .23746 L
.78481 .26121 L
.77215 .26121 L
F
.88608 .23746 m
.89873 .23746 L
.89873 .26121 L
.88608 .26121 L
F
.83544 .23746 m
.8481 .23746 L
.8481 .26121 L
.83544 .26121 L
F
.78481 .23746 m
.79747 .23746 L
.79747 .26121 L
.78481 .26121 L
F
.86076 .23746 m
.87342 .23746 L
.87342 .26121 L
.86076 .26121 L
F
.8481 .23746 m
.86076 .23746 L
.86076 .26121 L
.8481 .26121 L
F
.13924 .23746 m
.1519 .23746 L
.1519 .26121 L
.13924 .26121 L
F
.87342 .23746 m
.88608 .23746 L
.88608 .26121 L
.87342 .26121 L
F
.89873 .23746 m
.91139 .23746 L
.91139 .26121 L
.89873 .26121 L
F
.01266 .26121 m
.02532 .26121 L
.02532 .28496 L
.01266 .28496 L
F
.02532 .26121 m
.03797 .26121 L
.03797 .28496 L
.02532 .28496 L
F
.03797 .26121 m
.05063 .26121 L
.05063 .28496 L
.03797 .28496 L
F
.91139 .23746 m
.92405 .23746 L
.92405 .26121 L
.91139 .26121 L
F
.93671 .23746 m
.94937 .23746 L
.94937 .26121 L
.93671 .26121 L
F
.82278 .23746 m
.83544 .23746 L
.83544 .26121 L
.82278 .26121 L
F
.81013 .23746 m
.82278 .23746 L
.82278 .26121 L
.81013 .26121 L
F
.98734 .23746 m
1 .23746 L
1 .26121 L
.98734 .26121 L
F
0 .26121 m
.01266 .26121 L
.01266 .28496 L
0 .28496 L
F
.43038 .26121 m
.44304 .26121 L
.44304 .28496 L
.43038 .28496 L
F
.44304 .26121 m
.4557 .26121 L
.4557 .28496 L
.44304 .28496 L
F
.4557 .26121 m
.46835 .26121 L
.46835 .28496 L
.4557 .28496 L
F
.32911 .26121 m
.34177 .26121 L
.34177 .28496 L
.32911 .28496 L
F
.35443 .26121 m
.36709 .26121 L
.36709 .28496 L
.35443 .28496 L
F
.88608 .28496 m
.89873 .28496 L
.89873 .3087 L
.88608 .3087 L
F
.89873 .28496 m
.91139 .28496 L
.91139 .3087 L
.89873 .3087 L
F
.91139 .28496 m
.92405 .28496 L
.92405 .3087 L
.91139 .3087 L
F
.40506 .26121 m
.41772 .26121 L
.41772 .28496 L
.40506 .28496 L
F
.41772 .26121 m
.43038 .26121 L
.43038 .28496 L
.41772 .28496 L
F
.59494 .26121 m
.60759 .26121 L
.60759 .28496 L
.59494 .28496 L
F
.56962 .26121 m
.58228 .26121 L
.58228 .28496 L
.56962 .28496 L
F
.5443 .26121 m
.55696 .26121 L
.55696 .28496 L
.5443 .28496 L
F
.49367 .26121 m
.50633 .26121 L
.50633 .28496 L
.49367 .28496 L
F
.58228 .26121 m
.59494 .26121 L
.59494 .28496 L
.58228 .28496 L
F
.39241 .26121 m
.40506 .26121 L
.40506 .28496 L
.39241 .28496 L
F
.36709 .26121 m
.37975 .26121 L
.37975 .28496 L
.36709 .28496 L
F
.34177 .26121 m
.35443 .26121 L
.35443 .28496 L
.34177 .28496 L
F
.55696 .26121 m
.56962 .26121 L
.56962 .28496 L
.55696 .28496 L
F
.37975 .26121 m
.39241 .26121 L
.39241 .28496 L
.37975 .28496 L
F
.17722 .3087 m
.18987 .3087 L
.18987 .33245 L
.17722 .33245 L
F
.79747 .28496 m
.81013 .28496 L
.81013 .3087 L
.79747 .3087 L
F
.16456 .3087 m
.17722 .3087 L
.17722 .33245 L
.16456 .33245 L
F
.81013 .28496 m
.82278 .28496 L
.82278 .3087 L
.81013 .3087 L
F
.83544 .28496 m
.8481 .28496 L
.8481 .3087 L
.83544 .3087 L
F
.1519 .3087 m
.16456 .3087 L
.16456 .33245 L
.1519 .33245 L
F
.12658 .3087 m
.13924 .3087 L
.13924 .33245 L
.12658 .33245 L
F
.24051 .3087 m
.25316 .3087 L
.25316 .33245 L
.24051 .33245 L
F
.18987 .3087 m
.20253 .3087 L
.20253 .33245 L
.18987 .33245 L
F
.13924 .3087 m
.1519 .3087 L
.1519 .33245 L
.13924 .33245 L
F
.96203 .28496 m
.97468 .28496 L
.97468 .3087 L
.96203 .3087 L
F
.87342 .28496 m
.88608 .28496 L
.88608 .3087 L
.87342 .3087 L
F
.94937 .28496 m
.96203 .28496 L
.96203 .3087 L
.94937 .3087 L
F
.86076 .28496 m
.87342 .28496 L
.87342 .3087 L
.86076 .3087 L
F
.8481 .28496 m
.86076 .28496 L
.86076 .3087 L
.8481 .3087 L
F
.93671 .28496 m
.94937 .28496 L
.94937 .3087 L
.93671 .3087 L
F
.82278 .28496 m
.83544 .28496 L
.83544 .3087 L
.82278 .3087 L
F
.97468 .28496 m
.98734 .28496 L
.98734 .3087 L
.97468 .3087 L
F
.92405 .28496 m
.93671 .28496 L
.93671 .3087 L
.92405 .3087 L
F
.22785 .26121 m
.24051 .26121 L
.24051 .28496 L
.22785 .28496 L
F
.24051 .26121 m
.25316 .26121 L
.25316 .28496 L
.24051 .28496 L
F
.25316 .26121 m
.26582 .26121 L
.26582 .28496 L
.25316 .28496 L
F
.96203 .26121 m
.97468 .26121 L
.97468 .28496 L
.96203 .28496 L
F
.97468 .26121 m
.98734 .26121 L
.98734 .28496 L
.97468 .28496 L
F
.3038 .26121 m
.31646 .26121 L
.31646 .28496 L
.3038 .28496 L
F
.21519 .26121 m
.22785 .26121 L
.22785 .28496 L
.21519 .28496 L
F
.29114 .26121 m
.3038 .26121 L
.3038 .28496 L
.29114 .28496 L
F
.20253 .26121 m
.21519 .26121 L
.21519 .28496 L
.20253 .28496 L
F
.18987 .26121 m
.20253 .26121 L
.20253 .28496 L
.18987 .28496 L
F
.89873 .26121 m
.91139 .26121 L
.91139 .28496 L
.89873 .28496 L
F
.94937 .26121 m
.96203 .26121 L
.96203 .28496 L
.94937 .28496 L
F
.88608 .26121 m
.89873 .26121 L
.89873 .28496 L
.88608 .28496 L
F
.93671 .26121 m
.94937 .26121 L
.94937 .28496 L
.93671 .28496 L
F
.92405 .26121 m
.93671 .26121 L
.93671 .28496 L
.92405 .28496 L
F
.98734 .26121 m
1 .26121 L
1 .28496 L
.98734 .28496 L
F
.01266 .28496 m
.02532 .28496 L
.02532 .3087 L
.01266 .3087 L
F
.91139 .26121 m
.92405 .26121 L
.92405 .28496 L
.91139 .28496 L
F
0 .28496 m
.01266 .28496 L
.01266 .3087 L
0 .3087 L
F
.12658 .26121 m
.13924 .26121 L
.13924 .28496 L
.12658 .28496 L
F
.17722 .26121 m
.18987 .26121 L
.18987 .28496 L
.17722 .28496 L
F
.13924 .26121 m
.1519 .26121 L
.1519 .28496 L
.13924 .28496 L
F
.16456 .26121 m
.17722 .26121 L
.17722 .28496 L
.16456 .28496 L
F
.1519 .26121 m
.16456 .26121 L
.16456 .28496 L
.1519 .28496 L
F
.48101 .26121 m
.49367 .26121 L
.49367 .28496 L
.48101 .28496 L
F
.53165 .26121 m
.5443 .26121 L
.5443 .28496 L
.53165 .28496 L
F
.46835 .26121 m
.48101 .26121 L
.48101 .28496 L
.46835 .28496 L
F
.51899 .26121 m
.53165 .26121 L
.53165 .28496 L
.51899 .28496 L
F
.50633 .26121 m
.51899 .26121 L
.51899 .28496 L
.50633 .28496 L
F
.10127 .26121 m
.11392 .26121 L
.11392 .28496 L
.10127 .28496 L
F
.27848 .26121 m
.29114 .26121 L
.29114 .28496 L
.27848 .28496 L
F
.08861 .26121 m
.10127 .26121 L
.10127 .28496 L
.08861 .28496 L
F
.31646 .26121 m
.32911 .26121 L
.32911 .28496 L
.31646 .28496 L
F
.26582 .26121 m
.27848 .26121 L
.27848 .28496 L
.26582 .28496 L
F
.05063 .26121 m
.06329 .26121 L
.06329 .28496 L
.05063 .28496 L
F
.07595 .26121 m
.08861 .26121 L
.08861 .28496 L
.07595 .28496 L
F
.11392 .26121 m
.12658 .26121 L
.12658 .28496 L
.11392 .28496 L
F
.06329 .26121 m
.07595 .26121 L
.07595 .28496 L
.06329 .28496 L
F
.49367 .28496 m
.50633 .28496 L
.50633 .3087 L
.49367 .3087 L
F
.46835 .28496 m
.48101 .28496 L
.48101 .3087 L
.46835 .3087 L
F
.44304 .28496 m
.4557 .28496 L
.4557 .3087 L
.44304 .3087 L
F
.65823 .28496 m
.67089 .28496 L
.67089 .3087 L
.65823 .3087 L
F
.48101 .28496 m
.49367 .28496 L
.49367 .3087 L
.48101 .3087 L
F
.55696 .28496 m
.56962 .28496 L
.56962 .3087 L
.55696 .3087 L
F
.53165 .28496 m
.5443 .28496 L
.5443 .3087 L
.53165 .3087 L
F
.50633 .28496 m
.51899 .28496 L
.51899 .3087 L
.50633 .3087 L
F
.4557 .28496 m
.46835 .28496 L
.46835 .3087 L
.4557 .3087 L
F
.5443 .28496 m
.55696 .28496 L
.55696 .3087 L
.5443 .3087 L
F
.56962 .28496 m
.58228 .28496 L
.58228 .3087 L
.56962 .3087 L
F
.58228 .28496 m
.59494 .28496 L
.59494 .3087 L
.58228 .3087 L
F
.59494 .28496 m
.60759 .28496 L
.60759 .3087 L
.59494 .3087 L
F
.60759 .28496 m
.62025 .28496 L
.62025 .3087 L
.60759 .3087 L
F
.63291 .28496 m
.64557 .28496 L
.64557 .3087 L
.63291 .3087 L
F
.6962 .28496 m
.70886 .28496 L
.70886 .3087 L
.6962 .3087 L
F
.64557 .28496 m
.65823 .28496 L
.65823 .3087 L
.64557 .3087 L
F
.68354 .28496 m
.6962 .28496 L
.6962 .3087 L
.68354 .3087 L
F
.67089 .28496 m
.68354 .28496 L
.68354 .3087 L
.67089 .3087 L
F
.44304 .21372 m
.4557 .21372 L
.4557 .23746 L
.44304 .23746 L
F
.62025 .21372 m
.63291 .21372 L
.63291 .23746 L
.62025 .23746 L
F
.43038 .21372 m
.44304 .21372 L
.44304 .23746 L
.43038 .23746 L
F
.65823 .21372 m
.67089 .21372 L
.67089 .23746 L
.65823 .23746 L
F
.60759 .21372 m
.62025 .21372 L
.62025 .23746 L
.60759 .23746 L
F
.41772 .21372 m
.43038 .21372 L
.43038 .23746 L
.41772 .23746 L
F
.39241 .21372 m
.40506 .21372 L
.40506 .23746 L
.39241 .23746 L
F
.50633 .21372 m
.51899 .21372 L
.51899 .23746 L
.50633 .23746 L
F
.4557 .21372 m
.46835 .21372 L
.46835 .23746 L
.4557 .23746 L
F
.40506 .21372 m
.41772 .21372 L
.41772 .23746 L
.40506 .23746 L
F
.59494 .21372 m
.60759 .21372 L
.60759 .23746 L
.59494 .23746 L
F
.56962 .21372 m
.58228 .21372 L
.58228 .23746 L
.56962 .23746 L
F
.5443 .21372 m
.55696 .21372 L
.55696 .23746 L
.5443 .23746 L
F
.51899 .28496 m
.53165 .28496 L
.53165 .3087 L
.51899 .3087 L
F
.58228 .21372 m
.59494 .21372 L
.59494 .23746 L
.58228 .23746 L
F
.64557 .21372 m
.65823 .21372 L
.65823 .23746 L
.64557 .23746 L
F
.63291 .21372 m
.64557 .21372 L
.64557 .23746 L
.63291 .23746 L
F
.53165 .21372 m
.5443 .21372 L
.5443 .23746 L
.53165 .23746 L
F
.55696 .21372 m
.56962 .21372 L
.56962 .23746 L
.55696 .23746 L
F
.11392 .3087 m
.12658 .3087 L
.12658 .33245 L
.11392 .33245 L
F
.08861 .3087 m
.10127 .3087 L
.10127 .33245 L
.08861 .33245 L
F
.06329 .3087 m
.07595 .3087 L
.07595 .33245 L
.06329 .33245 L
F
.01266 .3087 m
.02532 .3087 L
.02532 .33245 L
.01266 .33245 L
F
.10127 .3087 m
.11392 .3087 L
.11392 .33245 L
.10127 .33245 L
F
.36709 .28496 m
.37975 .28496 L
.37975 .3087 L
.36709 .3087 L
F
.34177 .28496 m
.35443 .28496 L
.35443 .3087 L
.34177 .3087 L
F
.31646 .28496 m
.32911 .28496 L
.32911 .3087 L
.31646 .3087 L
F
.07595 .3087 m
.08861 .3087 L
.08861 .33245 L
.07595 .33245 L
F
.35443 .28496 m
.36709 .28496 L
.36709 .3087 L
.35443 .3087 L
F
.21519 .3087 m
.22785 .3087 L
.22785 .33245 L
.21519 .33245 L
F
.20253 .3087 m
.21519 .3087 L
.21519 .33245 L
.20253 .33245 L
F
.03797 .3087 m
.05063 .3087 L
.05063 .33245 L
.03797 .33245 L
F
.22785 .3087 m
.24051 .3087 L
.24051 .33245 L
.22785 .33245 L
F
.25316 .3087 m
.26582 .3087 L
.26582 .33245 L
.25316 .33245 L
F
0 .3087 m
.01266 .3087 L
.01266 .33245 L
0 .33245 L
F
.98734 .28496 m
1 .28496 L
1 .3087 L
.98734 .3087 L
F
.02532 .3087 m
.03797 .3087 L
.03797 .33245 L
.02532 .33245 L
F
.05063 .3087 m
.06329 .3087 L
.06329 .33245 L
.05063 .33245 L
F
.18987 .28496 m
.20253 .28496 L
.20253 .3087 L
.18987 .3087 L
F
.16456 .28496 m
.17722 .28496 L
.17722 .3087 L
.16456 .3087 L
F
.27848 .28496 m
.29114 .28496 L
.29114 .3087 L
.27848 .3087 L
F
.22785 .28496 m
.24051 .28496 L
.24051 .3087 L
.22785 .3087 L
F
.17722 .28496 m
.18987 .28496 L
.18987 .3087 L
.17722 .3087 L
F
.25316 .28496 m
.26582 .28496 L
.26582 .3087 L
.25316 .3087 L
F
.24051 .28496 m
.25316 .28496 L
.25316 .3087 L
.24051 .3087 L
F
.62025 .28496 m
.63291 .28496 L
.63291 .3087 L
.62025 .3087 L
F
.26582 .28496 m
.27848 .28496 L
.27848 .3087 L
.26582 .3087 L
F
.29114 .28496 m
.3038 .28496 L
.3038 .3087 L
.29114 .3087 L
F
.40506 .28496 m
.41772 .28496 L
.41772 .3087 L
.40506 .3087 L
F
.41772 .28496 m
.43038 .28496 L
.43038 .3087 L
.41772 .3087 L
F
.43038 .28496 m
.44304 .28496 L
.44304 .3087 L
.43038 .3087 L
F
.3038 .28496 m
.31646 .28496 L
.31646 .3087 L
.3038 .3087 L
F
.32911 .28496 m
.34177 .28496 L
.34177 .3087 L
.32911 .3087 L
F
.21519 .28496 m
.22785 .28496 L
.22785 .3087 L
.21519 .3087 L
F
.20253 .28496 m
.21519 .28496 L
.21519 .3087 L
.20253 .3087 L
F
.37975 .28496 m
.39241 .28496 L
.39241 .3087 L
.37975 .3087 L
F
.39241 .28496 m
.40506 .28496 L
.40506 .3087 L
.39241 .3087 L
F
.88608 1.51977 m
.89873 1.51977 L
.89873 1.54352 L
.88608 1.54352 L
F
.89873 1.51977 m
.91139 1.51977 L
.91139 1.54352 L
.89873 1.54352 L
F
.83544 1.51977 m
.8481 1.51977 L
.8481 1.54352 L
.83544 1.54352 L
F
.87342 1.51977 m
.88608 1.51977 L
.88608 1.54352 L
.87342 1.54352 L
F
.86076 1.51977 m
.87342 1.51977 L
.87342 1.54352 L
.86076 1.54352 L
F
.96203 1.51977 m
.97468 1.51977 L
.97468 1.54352 L
.96203 1.54352 L
F
.94937 1.51977 m
.96203 1.51977 L
.96203 1.54352 L
.94937 1.54352 L
F
.82278 1.51977 m
.83544 1.51977 L
.83544 1.54352 L
.82278 1.54352 L
F
.8481 1.51977 m
.86076 1.51977 L
.86076 1.54352 L
.8481 1.54352 L
F
.63291 1.54352 m
.64557 1.54352 L
.64557 1.56726 L
.63291 1.56726 L
F
.68354 1.54352 m
.6962 1.54352 L
.6962 1.56726 L
.68354 1.56726 L
F
.62025 1.54352 m
.63291 1.54352 L
.63291 1.56726 L
.62025 1.56726 L
F
.67089 1.54352 m
.68354 1.54352 L
.68354 1.56726 L
.67089 1.56726 L
F
.65823 1.54352 m
.67089 1.54352 L
.67089 1.56726 L
.65823 1.56726 L
F
.78481 1.54686 m
.78481 1.55539 L
.77848 1.55539 L
.77848 1.55536 L
F
.71519 1.54352 m
.72152 1.54352 L
.72152 1.55539 L
.71519 1.55539 L
F
.64557 1.54352 m
.65823 1.54352 L
.65823 1.56726 L
.64557 1.56726 L
F
.79114 1.54352 m
.79747 1.54352 L
.79747 1.55539 L
.79114 1.55539 L
F
.81013 1.51977 m
.82278 1.51977 L
.82278 1.54352 L
.81013 1.54352 L
F
.79114 1.51977 m
.79747 1.51977 L
.79747 1.53164 L
.79114 1.53164 L
F
.68354 1.51977 m
.6962 1.51977 L
.6962 1.54352 L
.68354 1.54352 L
F
.59494 1.51977 m
.60759 1.51977 L
.60759 1.54352 L
.59494 1.54352 L
F
.62025 1.51977 m
.63291 1.51977 L
.63291 1.54352 L
.62025 1.54352 L
F
.17722 1.54352 m
.18987 1.54352 L
.18987 1.56726 L
.17722 1.56726 L
F
.21519 1.54352 m
.22785 1.54352 L
.22785 1.56726 L
.21519 1.56726 L
F
.67089 1.51977 m
.68354 1.51977 L
.68354 1.54352 L
.67089 1.54352 L
F
.18987 1.54352 m
.20253 1.54352 L
.20253 1.56726 L
.18987 1.56726 L
F
.91139 1.51977 m
.92405 1.51977 L
.92405 1.54352 L
.91139 1.54352 L
F
.93671 1.51977 m
.94937 1.51977 L
.94937 1.54352 L
.93671 1.54352 L
F
.97468 1.51977 m
.98734 1.51977 L
.98734 1.54352 L
.97468 1.54352 L
F
.92405 1.51977 m
.93671 1.51977 L
.93671 1.54352 L
.92405 1.54352 L
F
.65823 1.51977 m
.67089 1.51977 L
.67089 1.54352 L
.65823 1.54352 L
F
.60759 1.51977 m
.62025 1.51977 L
.62025 1.54352 L
.60759 1.54352 L
F
.64557 1.51977 m
.65823 1.51977 L
.65823 1.54352 L
.64557 1.54352 L
F
.63291 1.51977 m
.64557 1.51977 L
.64557 1.54352 L
.63291 1.54352 L
F
.36709 1.54352 m
.37975 1.54352 L
.37975 1.56726 L
.36709 1.56726 L
F
.34177 1.54352 m
.35443 1.54352 L
.35443 1.56726 L
.34177 1.56726 L
F
.31646 1.54352 m
.32911 1.54352 L
.32911 1.56726 L
.31646 1.56726 L
F
.56962 1.54352 m
.58228 1.54352 L
.58228 1.56726 L
.56962 1.56726 L
F
.35443 1.54352 m
.36709 1.54352 L
.36709 1.56726 L
.35443 1.56726 L
F
.43038 1.54352 m
.44304 1.54352 L
.44304 1.56726 L
.43038 1.56726 L
F
.41772 1.54352 m
.43038 1.54352 L
.43038 1.56726 L
.41772 1.56726 L
F
.3038 1.54352 m
.31646 1.54352 L
.31646 1.56726 L
.3038 1.56726 L
F
.32911 1.54352 m
.34177 1.54352 L
.34177 1.56726 L
.32911 1.56726 L
F
.48101 1.54352 m
.49367 1.54352 L
.49367 1.56726 L
.48101 1.56726 L
F
.4557 1.54352 m
.46835 1.54352 L
.46835 1.56726 L
.4557 1.56726 L
F
.59494 1.54352 m
.60759 1.54352 L
.60759 1.56726 L
.59494 1.56726 L
F
.51899 1.54352 m
.53165 1.54352 L
.53165 1.56726 L
.51899 1.56726 L
F
.46835 1.54352 m
.48101 1.54352 L
.48101 1.56726 L
.46835 1.56726 L
F
.55696 1.54352 m
.56962 1.54352 L
.56962 1.56726 L
.55696 1.56726 L
F
.5443 1.54352 m
.55696 1.54352 L
.55696 1.56726 L
.5443 1.56726 L
F
.58228 1.54352 m
.59494 1.54352 L
.59494 1.56726 L
.58228 1.56726 L
F
.60759 1.54352 m
.62025 1.54352 L
.62025 1.56726 L
.60759 1.56726 L
F
.94937 1.54352 m
.96203 1.54352 L
.96203 1.56726 L
.94937 1.56726 L
F
.83544 1.54352 m
.8481 1.54352 L
.8481 1.56726 L
.83544 1.56726 L
F
.93671 1.54352 m
.94937 1.54352 L
.94937 1.56726 L
.93671 1.56726 L
F
.82278 1.54352 m
.83544 1.54352 L
.83544 1.56726 L
.82278 1.56726 L
F
.81013 1.54352 m
.82278 1.54352 L
.82278 1.56726 L
.81013 1.56726 L
F
.89873 1.54352 m
.91139 1.54352 L
.91139 1.56726 L
.89873 1.56726 L
F
.92405 1.54352 m
.93671 1.54352 L
.93671 1.56726 L
.92405 1.56726 L
F
.96203 1.54352 m
.97468 1.54352 L
.97468 1.56726 L
.96203 1.56726 L
F
.91139 1.54352 m
.92405 1.54352 L
.92405 1.56726 L
.91139 1.56726 L
F
.37975 1.54352 m
.39241 1.54352 L
.39241 1.56726 L
.37975 1.56726 L
F
.40506 1.54352 m
.41772 1.54352 L
.41772 1.56726 L
.40506 1.56726 L
F
.44304 1.54352 m
.4557 1.54352 L
.4557 1.56726 L
.44304 1.56726 L
F
.39241 1.54352 m
.40506 1.54352 L
.40506 1.56726 L
.39241 1.56726 L
F
.88608 1.54352 m
.89873 1.54352 L
.89873 1.56726 L
.88608 1.56726 L
F
.87342 1.54352 m
.88608 1.54352 L
.88608 1.56726 L
.87342 1.56726 L
F
.86076 1.54352 m
.87342 1.54352 L
.87342 1.56726 L
.86076 1.56726 L
F
.8481 1.54352 m
.86076 1.54352 L
.86076 1.56726 L
.8481 1.56726 L
F
.8481 1.44853 m
.86076 1.44853 L
.86076 1.47228 L
.8481 1.47228 L
F
.43038 1.44853 m
.44304 1.44853 L
.44304 1.47228 L
.43038 1.47228 L
F
.83544 1.44853 m
.8481 1.44853 L
.8481 1.47228 L
.83544 1.47228 L
F
.41772 1.44853 m
.43038 1.44853 L
.43038 1.47228 L
.41772 1.47228 L
F
.40506 1.44853 m
.41772 1.44853 L
.41772 1.47228 L
.40506 1.47228 L
F
.81013 1.44853 m
.82278 1.44853 L
.82278 1.47228 L
.81013 1.47228 L
F
.79747 1.44853 m
.81013 1.44853 L
.81013 1.47228 L
.79747 1.47228 L
F
.87342 1.44853 m
.88608 1.44853 L
.88608 1.47228 L
.87342 1.47228 L
F
.86076 1.44853 m
.87342 1.44853 L
.87342 1.47228 L
.86076 1.47228 L
F
.34177 1.44853 m
.35443 1.44853 L
.35443 1.47228 L
.34177 1.47228 L
F
.39241 1.44853 m
.40506 1.44853 L
.40506 1.47228 L
.39241 1.47228 L
F
.32911 1.44853 m
.34177 1.44853 L
.34177 1.47228 L
.32911 1.47228 L
F
.37975 1.44853 m
.39241 1.44853 L
.39241 1.47228 L
.37975 1.47228 L
F
.36709 1.44853 m
.37975 1.44853 L
.37975 1.47228 L
.36709 1.47228 L
F
.44304 1.44853 m
.4557 1.44853 L
.4557 1.47228 L
.44304 1.47228 L
F
.46835 1.44853 m
.48101 1.44853 L
.48101 1.47228 L
.46835 1.47228 L
F
.35443 1.44853 m
.36709 1.44853 L
.36709 1.47228 L
.35443 1.47228 L
F
.4557 1.44853 m
.46835 1.44853 L
.46835 1.47228 L
.4557 1.47228 L
F
.64557 1.44853 m
.65823 1.44853 L
.65823 1.47228 L
.64557 1.47228 L
F
.65823 1.44853 m
.67089 1.44853 L
.67089 1.47228 L
.65823 1.47228 L
F
.67089 1.44853 m
.68354 1.44853 L
.68354 1.47228 L
.67089 1.47228 L
F
.68354 1.44853 m
.6962 1.44853 L
.6962 1.47228 L
.68354 1.47228 L
F
.70886 1.44853 m
.72152 1.44853 L
.72152 1.47228 L
.70886 1.47228 L
F
.78481 1.44853 m
.79747 1.44853 L
.79747 1.47228 L
.78481 1.47228 L
F
.73418 1.44853 m
.74684 1.44853 L
.74684 1.47228 L
.73418 1.47228 L
F
.77215 1.44853 m
.78481 1.44853 L
.78481 1.47228 L
.77215 1.47228 L
F
.75949 1.44853 m
.77215 1.44853 L
.77215 1.47228 L
.75949 1.47228 L
F
.92405 1.44853 m
.93671 1.44853 L
.93671 1.47228 L
.92405 1.47228 L
F
.94937 1.44853 m
.96203 1.44853 L
.96203 1.47228 L
.94937 1.47228 L
F
.82278 1.44853 m
.83544 1.44853 L
.83544 1.47228 L
.82278 1.47228 L
F
.93671 1.44853 m
.94937 1.44853 L
.94937 1.47228 L
.93671 1.47228 L
F
.91139 1.44853 m
.92405 1.44853 L
.92405 1.47228 L
.91139 1.47228 L
F
.6962 1.44853 m
.70886 1.44853 L
.70886 1.47228 L
.6962 1.47228 L
F
.89873 1.44853 m
.91139 1.44853 L
.91139 1.47228 L
.89873 1.47228 L
F
.88608 1.44853 m
.89873 1.44853 L
.89873 1.47228 L
.88608 1.47228 L
F
.05063 1.54352 m
.06329 1.54352 L
.06329 1.56726 L
.05063 1.56726 L
F
.02532 1.54352 m
.03797 1.54352 L
.03797 1.56726 L
.02532 1.56726 L
F
0 1.54352 m
.01266 1.54352 L
.01266 1.56726 L
0 1.56726 L
F
.25316 1.54352 m
.26582 1.54352 L
.26582 1.56726 L
.25316 1.56726 L
F
.03797 1.54352 m
.05063 1.54352 L
.05063 1.56726 L
.03797 1.56726 L
F
.11392 1.54352 m
.12658 1.54352 L
.12658 1.56726 L
.11392 1.56726 L
F
.10127 1.54352 m
.11392 1.54352 L
.11392 1.56726 L
.10127 1.56726 L
F
.98734 1.51977 m
1 1.51977 L
1 1.54352 L
.98734 1.54352 L
F
.01266 1.54352 m
.02532 1.54352 L
.02532 1.56726 L
.01266 1.56726 L
F
.16456 1.54352 m
.17722 1.54352 L
.17722 1.56726 L
.16456 1.56726 L
F
.13924 1.54352 m
.1519 1.54352 L
.1519 1.56726 L
.13924 1.56726 L
F
.27848 1.54352 m
.29114 1.54352 L
.29114 1.56726 L
.27848 1.56726 L
F
.20253 1.54352 m
.21519 1.54352 L
.21519 1.56726 L
.20253 1.56726 L
F
.1519 1.54352 m
.16456 1.54352 L
.16456 1.56726 L
.1519 1.56726 L
F
.24051 1.54352 m
.25316 1.54352 L
.25316 1.56726 L
.24051 1.56726 L
F
.22785 1.54352 m
.24051 1.54352 L
.24051 1.56726 L
.22785 1.56726 L
F
.26582 1.54352 m
.27848 1.54352 L
.27848 1.56726 L
.26582 1.56726 L
F
.29114 1.54352 m
.3038 1.54352 L
.3038 1.56726 L
.29114 1.56726 L
F
.62025 1.44853 m
.63291 1.44853 L
.63291 1.47228 L
.62025 1.47228 L
F
.50633 1.44853 m
.51899 1.44853 L
.51899 1.47228 L
.50633 1.47228 L
F
.60759 1.44853 m
.62025 1.44853 L
.62025 1.47228 L
.60759 1.47228 L
F
.49367 1.44853 m
.50633 1.44853 L
.50633 1.47228 L
.49367 1.47228 L
F
.48101 1.44853 m
.49367 1.44853 L
.49367 1.47228 L
.48101 1.47228 L
F
.56962 1.44853 m
.58228 1.44853 L
.58228 1.47228 L
.56962 1.47228 L
F
.59494 1.44853 m
.60759 1.44853 L
.60759 1.47228 L
.59494 1.47228 L
F
.63291 1.44853 m
.64557 1.44853 L
.64557 1.47228 L
.63291 1.47228 L
F
.58228 1.44853 m
.59494 1.44853 L
.59494 1.47228 L
.58228 1.47228 L
F
.06329 1.54352 m
.07595 1.54352 L
.07595 1.56726 L
.06329 1.56726 L
F
.08861 1.54352 m
.10127 1.54352 L
.10127 1.56726 L
.08861 1.56726 L
F
.12658 1.54352 m
.13924 1.54352 L
.13924 1.56726 L
.12658 1.56726 L
F
.07595 1.54352 m
.08861 1.54352 L
.08861 1.56726 L
.07595 1.56726 L
F
.55696 1.44853 m
.56962 1.44853 L
.56962 1.47228 L
.55696 1.47228 L
F
.5443 1.44853 m
.55696 1.44853 L
.55696 1.47228 L
.5443 1.47228 L
F
.53165 1.44853 m
.5443 1.44853 L
.5443 1.47228 L
.53165 1.47228 L
F
.51899 1.44853 m
.53165 1.44853 L
.53165 1.47228 L
.51899 1.47228 L
F
.21519 1.51977 m
.22785 1.51977 L
.22785 1.54352 L
.21519 1.54352 L
F
.26582 1.51977 m
.27848 1.51977 L
.27848 1.54352 L
.26582 1.54352 L
F
.20253 1.51977 m
.21519 1.51977 L
.21519 1.54352 L
.20253 1.54352 L
F
.25316 1.51977 m
.26582 1.51977 L
.26582 1.54352 L
.25316 1.54352 L
F
.24051 1.51977 m
.25316 1.51977 L
.25316 1.54352 L
.24051 1.54352 L
F
0 1.51977 m
.01266 1.51977 L
.01266 1.54352 L
0 1.54352 L
F
.02532 1.51977 m
.03797 1.51977 L
.03797 1.54352 L
.02532 1.54352 L
F
.22785 1.51977 m
.24051 1.51977 L
.24051 1.54352 L
.22785 1.54352 L
F
.01266 1.51977 m
.02532 1.51977 L
.02532 1.54352 L
.01266 1.54352 L
F
.1519 1.51977 m
.16456 1.51977 L
.16456 1.54352 L
.1519 1.54352 L
F
.16456 1.51977 m
.17722 1.51977 L
.17722 1.54352 L
.16456 1.54352 L
F
.18987 1.51977 m
.20253 1.51977 L
.20253 1.54352 L
.18987 1.54352 L
F
.74684 1.82847 m
.75949 1.82847 L
.75949 1.85222 L
.74684 1.85222 L
F
.77215 1.82847 m
.78481 1.82847 L
.78481 1.85222 L
.77215 1.85222 L
F
.11392 1.51977 m
.12658 1.51977 L
.12658 1.54352 L
.11392 1.54352 L
F
.13924 1.51977 m
.1519 1.51977 L
.1519 1.54352 L
.13924 1.54352 L
F
.17722 1.51977 m
.18987 1.51977 L
.18987 1.54352 L
.17722 1.54352 L
F
.12658 1.51977 m
.13924 1.51977 L
.13924 1.54352 L
.12658 1.54352 L
F
.50633 1.51977 m
.51899 1.51977 L
.51899 1.54352 L
.50633 1.54352 L
F
.46835 1.51977 m
.48101 1.51977 L
.48101 1.54352 L
.46835 1.54352 L
F
.49367 1.51977 m
.50633 1.51977 L
.50633 1.54352 L
.49367 1.54352 L
F
.06329 1.51977 m
.07595 1.51977 L
.07595 1.54352 L
.06329 1.54352 L
F
.48101 1.51977 m
.49367 1.51977 L
.49367 1.54352 L
.48101 1.54352 L
F
.4557 1.51977 m
.46835 1.51977 L
.46835 1.54352 L
.4557 1.54352 L
F
.56962 1.51977 m
.58228 1.51977 L
.58228 1.54352 L
.56962 1.54352 L
F
.44304 1.51977 m
.4557 1.51977 L
.4557 1.54352 L
.44304 1.54352 L
F
.43038 1.51977 m
.44304 1.51977 L
.44304 1.54352 L
.43038 1.54352 L
F
.98734 1.49602 m
1 1.49602 L
1 1.51977 L
.98734 1.51977 L
F
.08861 1.51977 m
.10127 1.51977 L
.10127 1.54352 L
.08861 1.54352 L
F
.97468 1.49602 m
.98734 1.49602 L
.98734 1.51977 L
.97468 1.51977 L
F
.96203 1.49602 m
.97468 1.49602 L
.97468 1.51977 L
.96203 1.51977 L
F
.05063 1.51977 m
.06329 1.51977 L
.06329 1.54352 L
.05063 1.54352 L
F
.03797 1.51977 m
.05063 1.51977 L
.05063 1.54352 L
.03797 1.54352 L
F
.07595 1.51977 m
.08861 1.51977 L
.08861 1.54352 L
.07595 1.54352 L
F
.10127 1.51977 m
.11392 1.51977 L
.11392 1.54352 L
.10127 1.54352 L
F
.4557 1.82847 m
.46835 1.82847 L
.46835 1.85222 L
.4557 1.85222 L
F
.43038 1.82847 m
.44304 1.82847 L
.44304 1.85222 L
.43038 1.85222 L
F
.87342 1.82847 m
.88608 1.82847 L
.88608 1.85222 L
.87342 1.85222 L
F
.49367 1.82847 m
.50633 1.82847 L
.50633 1.85222 L
.49367 1.85222 L
F
.44304 1.82847 m
.4557 1.82847 L
.4557 1.85222 L
.44304 1.85222 L
F
.83544 1.82847 m
.8481 1.82847 L
.8481 1.85222 L
.83544 1.85222 L
F
.82278 1.82847 m
.83544 1.82847 L
.83544 1.85222 L
.82278 1.85222 L
F
.86076 1.82847 m
.87342 1.82847 L
.87342 1.85222 L
.86076 1.85222 L
F
.88608 1.82847 m
.89873 1.82847 L
.89873 1.85222 L
.88608 1.85222 L
F
.41772 1.82847 m
.43038 1.82847 L
.43038 1.85222 L
.41772 1.85222 L
F
.39241 1.82847 m
.40506 1.82847 L
.40506 1.85222 L
.39241 1.85222 L
F
.36709 1.82847 m
.37975 1.82847 L
.37975 1.85222 L
.36709 1.85222 L
F
.62025 1.82847 m
.63291 1.82847 L
.63291 1.85222 L
.62025 1.85222 L
F
.40506 1.82847 m
.41772 1.82847 L
.41772 1.85222 L
.40506 1.85222 L
F
.48101 1.82847 m
.49367 1.82847 L
.49367 1.85222 L
.48101 1.85222 L
F
.46835 1.82847 m
.48101 1.82847 L
.48101 1.85222 L
.46835 1.85222 L
F
.35443 1.82847 m
.36709 1.82847 L
.36709 1.85222 L
.35443 1.85222 L
F
.37975 1.82847 m
.39241 1.82847 L
.39241 1.85222 L
.37975 1.85222 L
F
.67089 1.82847 m
.68354 1.82847 L
.68354 1.85222 L
.67089 1.85222 L
F
.68354 1.82847 m
.6962 1.82847 L
.6962 1.85222 L
.68354 1.85222 L
F
.6962 1.82847 m
.70886 1.82847 L
.70886 1.85222 L
.6962 1.85222 L
F
.70886 1.82847 m
.72152 1.82847 L
.72152 1.85222 L
.70886 1.85222 L
F
.73418 1.82847 m
.74684 1.82847 L
.74684 1.85222 L
.73418 1.85222 L
F
.81013 1.82847 m
.82278 1.82847 L
.82278 1.85222 L
.81013 1.85222 L
F
.75949 1.82847 m
.77215 1.82847 L
.77215 1.85222 L
.75949 1.85222 L
F
.79747 1.82847 m
.81013 1.82847 L
.81013 1.85222 L
.79747 1.85222 L
F
.78481 1.82847 m
.79747 1.82847 L
.79747 1.85222 L
.78481 1.85222 L
F
.93671 1.82847 m
.94937 1.82847 L
.94937 1.85222 L
.93671 1.85222 L
F
.96203 1.82847 m
.97468 1.82847 L
.97468 1.85222 L
.96203 1.85222 L
F
.8481 1.82847 m
.86076 1.82847 L
.86076 1.85222 L
.8481 1.85222 L
F
.94937 1.82847 m
.96203 1.82847 L
.96203 1.85222 L
.94937 1.85222 L
F
.92405 1.82847 m
.93671 1.82847 L
.93671 1.85222 L
.92405 1.85222 L
F
.72152 1.82847 m
.73418 1.82847 L
.73418 1.85222 L
.72152 1.85222 L
F
.91139 1.82847 m
.92405 1.82847 L
.92405 1.85222 L
.91139 1.85222 L
F
.89873 1.82847 m
.91139 1.82847 L
.91139 1.85222 L
.89873 1.85222 L
F
.81013 1.49602 m
.82278 1.49602 L
.82278 1.51977 L
.81013 1.51977 L
F
.86076 1.49602 m
.87342 1.49602 L
.87342 1.51977 L
.86076 1.51977 L
F
.79747 1.49602 m
.81013 1.49602 L
.81013 1.51977 L
.79747 1.51977 L
F
.83544 1.49602 m
.8481 1.49602 L
.8481 1.51977 L
.83544 1.51977 L
F
.87342 1.49602 m
.88608 1.49602 L
.88608 1.51977 L
.87342 1.51977 L
F
.92405 1.49602 m
.93671 1.49602 L
.93671 1.51977 L
.92405 1.51977 L
F
.94937 1.49602 m
.96203 1.49602 L
.96203 1.51977 L
.94937 1.51977 L
F
.82278 1.49602 m
.83544 1.49602 L
.83544 1.51977 L
.82278 1.51977 L
F
.93671 1.49602 m
.94937 1.49602 L
.94937 1.51977 L
.93671 1.51977 L
F
.36709 1.49602 m
.37975 1.49602 L
.37975 1.51977 L
.36709 1.51977 L
F
.37975 1.49602 m
.39241 1.49602 L
.39241 1.51977 L
.37975 1.51977 L
F
.39241 1.49602 m
.40506 1.49602 L
.40506 1.51977 L
.39241 1.51977 L
F
.25316 1.49602 m
.26582 1.49602 L
.26582 1.51977 L
.25316 1.51977 L
F
.27848 1.49602 m
.29114 1.49602 L
.29114 1.51977 L
.27848 1.51977 L
F
.35443 1.49602 m
.36709 1.49602 L
.36709 1.51977 L
.35443 1.51977 L
F
.8481 1.49602 m
.86076 1.49602 L
.86076 1.51977 L
.8481 1.51977 L
F
.34177 1.49602 m
.35443 1.49602 L
.35443 1.51977 L
.34177 1.51977 L
F
.32911 1.49602 m
.34177 1.49602 L
.34177 1.51977 L
.32911 1.51977 L
F
.65823 1.49602 m
.67089 1.49602 L
.67089 1.51977 L
.65823 1.51977 L
F
.71519 1.49602 m
.72152 1.49602 L
.72152 1.5079 L
.71519 1.5079 L
F
.64557 1.49602 m
.65823 1.49602 L
.65823 1.51977 L
.64557 1.51977 L
F
.59494 1.49602 m
.60759 1.49602 L
.60759 1.51977 L
.59494 1.51977 L
F
.68354 1.49602 m
.6962 1.49602 L
.6962 1.51977 L
.68354 1.51977 L
F
.49367 1.54352 m
.50633 1.54352 L
.50633 1.56726 L
.49367 1.56726 L
F
.53165 1.54352 m
.5443 1.54352 L
.5443 1.56726 L
.53165 1.56726 L
F
.67089 1.49602 m
.68354 1.49602 L
.68354 1.51977 L
.67089 1.51977 L
F
.50633 1.54352 m
.51899 1.54352 L
.51899 1.56726 L
.50633 1.56726 L
F
.91139 1.49602 m
.92405 1.49602 L
.92405 1.51977 L
.91139 1.51977 L
F
.62025 1.49602 m
.63291 1.49602 L
.63291 1.51977 L
.62025 1.51977 L
F
.89873 1.49602 m
.91139 1.49602 L
.91139 1.51977 L
.89873 1.51977 L
F
.88608 1.49602 m
.89873 1.49602 L
.89873 1.51977 L
.88608 1.51977 L
F
.58228 1.49602 m
.59494 1.49602 L
.59494 1.51977 L
.58228 1.51977 L
F
.56962 1.49602 m
.58228 1.49602 L
.58228 1.51977 L
.56962 1.51977 L
F
.60759 1.49602 m
.62025 1.49602 L
.62025 1.51977 L
.60759 1.51977 L
F
.63291 1.49602 m
.64557 1.49602 L
.64557 1.51977 L
.63291 1.51977 L
F
.40506 1.51977 m
.41772 1.51977 L
.41772 1.54352 L
.40506 1.54352 L
F
.3038 1.51977 m
.31646 1.51977 L
.31646 1.54352 L
.3038 1.54352 L
F
.39241 1.51977 m
.40506 1.51977 L
.40506 1.54352 L
.39241 1.54352 L
F
.29114 1.51977 m
.3038 1.51977 L
.3038 1.54352 L
.29114 1.54352 L
F
.27848 1.51977 m
.29114 1.51977 L
.29114 1.54352 L
.27848 1.54352 L
F
.35443 1.51977 m
.36709 1.51977 L
.36709 1.54352 L
.35443 1.54352 L
F
.37975 1.51977 m
.39241 1.51977 L
.39241 1.54352 L
.37975 1.54352 L
F
.41772 1.51977 m
.43038 1.51977 L
.43038 1.54352 L
.41772 1.54352 L
F
.36709 1.51977 m
.37975 1.51977 L
.37975 1.54352 L
.36709 1.54352 L
F
.53165 1.51977 m
.5443 1.51977 L
.5443 1.54352 L
.53165 1.54352 L
F
.51899 1.51977 m
.53165 1.51977 L
.53165 1.54352 L
.51899 1.54352 L
F
.55696 1.51977 m
.56962 1.51977 L
.56962 1.54352 L
.55696 1.54352 L
F
.58228 1.51977 m
.59494 1.51977 L
.59494 1.54352 L
.58228 1.54352 L
F
.31646 1.51977 m
.32911 1.51977 L
.32911 1.54352 L
.31646 1.54352 L
F
.34177 1.51977 m
.35443 1.51977 L
.35443 1.54352 L
.34177 1.54352 L
F
.5443 1.51977 m
.55696 1.51977 L
.55696 1.54352 L
.5443 1.54352 L
F
.32911 1.51977 m
.34177 1.51977 L
.34177 1.54352 L
.32911 1.54352 L
F
.49367 1.49602 m
.50633 1.49602 L
.50633 1.51977 L
.49367 1.51977 L
F
.50633 1.49602 m
.51899 1.49602 L
.51899 1.51977 L
.50633 1.51977 L
F
.51899 1.49602 m
.53165 1.49602 L
.53165 1.51977 L
.51899 1.51977 L
F
.53165 1.49602 m
.5443 1.49602 L
.5443 1.51977 L
.53165 1.51977 L
F
.55696 1.49602 m
.56962 1.49602 L
.56962 1.51977 L
.55696 1.51977 L
F
.31646 1.49602 m
.32911 1.49602 L
.32911 1.51977 L
.31646 1.51977 L
F
.26582 1.49602 m
.27848 1.49602 L
.27848 1.51977 L
.26582 1.51977 L
F
.3038 1.49602 m
.31646 1.49602 L
.31646 1.51977 L
.3038 1.51977 L
F
.29114 1.49602 m
.3038 1.49602 L
.3038 1.51977 L
.29114 1.51977 L
F
.48101 1.49602 m
.49367 1.49602 L
.49367 1.51977 L
.48101 1.51977 L
F
.46835 1.49602 m
.48101 1.49602 L
.48101 1.51977 L
.46835 1.51977 L
F
.4557 1.49602 m
.46835 1.49602 L
.46835 1.51977 L
.4557 1.51977 L
F
.44304 1.49602 m
.4557 1.49602 L
.4557 1.51977 L
.44304 1.51977 L
F
.43038 1.49602 m
.44304 1.49602 L
.44304 1.51977 L
.43038 1.51977 L
F
.5443 1.49602 m
.55696 1.49602 L
.55696 1.51977 L
.5443 1.51977 L
F
.41772 1.49602 m
.43038 1.49602 L
.43038 1.51977 L
.41772 1.51977 L
F
.40506 1.49602 m
.41772 1.49602 L
.41772 1.51977 L
.40506 1.51977 L
F
.40506 1.6385 m
.41772 1.6385 L
.41772 1.66225 L
.40506 1.66225 L
F
.41772 1.6385 m
.43038 1.6385 L
.43038 1.66225 L
.41772 1.66225 L
F
.43038 1.6385 m
.44304 1.6385 L
.44304 1.66225 L
.43038 1.66225 L
F
.27848 1.6385 m
.29114 1.6385 L
.29114 1.66225 L
.27848 1.66225 L
F
.29114 1.6385 m
.3038 1.6385 L
.3038 1.66225 L
.29114 1.66225 L
F
.3038 1.6385 m
.31646 1.6385 L
.31646 1.66225 L
.3038 1.66225 L
F
.17722 1.6385 m
.18987 1.6385 L
.18987 1.66225 L
.17722 1.66225 L
F
.39241 1.6385 m
.40506 1.6385 L
.40506 1.66225 L
.39241 1.66225 L
F
.16456 1.6385 m
.17722 1.6385 L
.17722 1.66225 L
.16456 1.66225 L
F
.37975 1.6385 m
.39241 1.6385 L
.39241 1.66225 L
.37975 1.66225 L
F
.36709 1.6385 m
.37975 1.6385 L
.37975 1.66225 L
.36709 1.66225 L
F
.36709 1.47228 m
.37975 1.47228 L
.37975 1.49602 L
.36709 1.49602 L
F
.41772 1.47228 m
.43038 1.47228 L
.43038 1.49602 L
.41772 1.49602 L
F
.35443 1.47228 m
.36709 1.47228 L
.36709 1.49602 L
.35443 1.49602 L
F
.40506 1.47228 m
.41772 1.47228 L
.41772 1.49602 L
.40506 1.49602 L
F
.39241 1.47228 m
.40506 1.47228 L
.40506 1.49602 L
.39241 1.49602 L
F
.35443 1.6385 m
.36709 1.6385 L
.36709 1.66225 L
.35443 1.66225 L
F
.31646 1.6385 m
.32911 1.6385 L
.32911 1.66225 L
.31646 1.66225 L
F
.34177 1.6385 m
.35443 1.6385 L
.35443 1.66225 L
.34177 1.66225 L
F
.37975 1.47228 m
.39241 1.47228 L
.39241 1.49602 L
.37975 1.49602 L
F
.32911 1.6385 m
.34177 1.6385 L
.34177 1.66225 L
.32911 1.66225 L
F
.60759 1.6385 m
.62025 1.6385 L
.62025 1.66225 L
.60759 1.66225 L
F
.65823 1.6385 m
.67089 1.6385 L
.67089 1.66225 L
.65823 1.66225 L
F
.59494 1.6385 m
.60759 1.6385 L
.60759 1.66225 L
.59494 1.66225 L
F
.63291 1.6385 m
.64557 1.6385 L
.64557 1.66225 L
.63291 1.66225 L
F
.64557 1.6385 m
.65823 1.6385 L
.65823 1.66225 L
.64557 1.66225 L
F
.67089 1.6385 m
.68354 1.6385 L
.68354 1.66225 L
.67089 1.66225 L
F
.74684 1.6385 m
.75949 1.6385 L
.75949 1.66225 L
.74684 1.66225 L
F
.72152 1.6385 m
.73418 1.6385 L
.73418 1.66225 L
.72152 1.66225 L
F
.6962 1.6385 m
.70886 1.6385 L
.70886 1.66225 L
.6962 1.66225 L
F
.62025 1.6385 m
.63291 1.6385 L
.63291 1.66225 L
.62025 1.66225 L
F
.73418 1.6385 m
.74684 1.6385 L
.74684 1.66225 L
.73418 1.66225 L
F
.1519 1.6385 m
.16456 1.6385 L
.16456 1.66225 L
.1519 1.66225 L
F
.12658 1.6385 m
.13924 1.6385 L
.13924 1.66225 L
.12658 1.66225 L
F
.25316 1.6385 m
.26582 1.6385 L
.26582 1.66225 L
.25316 1.66225 L
F
.18987 1.6385 m
.20253 1.6385 L
.20253 1.66225 L
.18987 1.66225 L
F
.13924 1.6385 m
.1519 1.6385 L
.1519 1.66225 L
.13924 1.66225 L
F
.20253 1.6385 m
.21519 1.6385 L
.21519 1.66225 L
.20253 1.66225 L
F
.21519 1.6385 m
.22785 1.6385 L
.22785 1.66225 L
.21519 1.66225 L
F
.22785 1.6385 m
.24051 1.6385 L
.24051 1.66225 L
.22785 1.66225 L
F
.24051 1.6385 m
.25316 1.6385 L
.25316 1.66225 L
.24051 1.66225 L
F
.26582 1.6385 m
.27848 1.6385 L
.27848 1.66225 L
.26582 1.66225 L
F
.97468 1.44853 m
.98734 1.44853 L
.98734 1.47228 L
.97468 1.47228 L
F
.02532 1.47228 m
.03797 1.47228 L
.03797 1.49602 L
.02532 1.49602 L
F
.96203 1.44853 m
.97468 1.44853 L
.97468 1.47228 L
.96203 1.47228 L
F
.01266 1.47228 m
.02532 1.47228 L
.02532 1.49602 L
.01266 1.49602 L
F
.22785 1.47228 m
.24051 1.47228 L
.24051 1.49602 L
.22785 1.49602 L
F
0 1.47228 m
.01266 1.47228 L
.01266 1.49602 L
0 1.49602 L
F
.10127 1.47228 m
.11392 1.47228 L
.11392 1.49602 L
.10127 1.49602 L
F
.07595 1.47228 m
.08861 1.47228 L
.08861 1.49602 L
.07595 1.49602 L
F
.05063 1.47228 m
.06329 1.47228 L
.06329 1.49602 L
.05063 1.49602 L
F
.98734 1.44853 m
1 1.44853 L
1 1.47228 L
.98734 1.47228 L
F
.08861 1.47228 m
.10127 1.47228 L
.10127 1.49602 L
.08861 1.49602 L
F
.11392 1.47228 m
.12658 1.47228 L
.12658 1.49602 L
.11392 1.49602 L
F
.12658 1.47228 m
.13924 1.47228 L
.13924 1.49602 L
.12658 1.49602 L
F
.13924 1.47228 m
.1519 1.47228 L
.1519 1.49602 L
.13924 1.49602 L
F
.18987 1.47228 m
.20253 1.47228 L
.20253 1.49602 L
.18987 1.49602 L
F
.17722 1.47228 m
.18987 1.47228 L
.18987 1.49602 L
.17722 1.49602 L
F
.21519 1.47228 m
.22785 1.47228 L
.22785 1.49602 L
.21519 1.49602 L
F
.26582 1.47228 m
.27848 1.47228 L
.27848 1.49602 L
.26582 1.49602 L
F
.20253 1.47228 m
.21519 1.47228 L
.21519 1.49602 L
.20253 1.49602 L
F
.25316 1.47228 m
.26582 1.47228 L
.26582 1.49602 L
.25316 1.49602 L
F
.24051 1.47228 m
.25316 1.47228 L
.25316 1.49602 L
.24051 1.49602 L
F
.53165 1.47228 m
.5443 1.47228 L
.5443 1.49602 L
.53165 1.49602 L
F
.50633 1.47228 m
.51899 1.47228 L
.51899 1.49602 L
.50633 1.49602 L
F
.32911 1.47228 m
.34177 1.47228 L
.34177 1.49602 L
.32911 1.49602 L
F
.56962 1.47228 m
.58228 1.47228 L
.58228 1.49602 L
.56962 1.49602 L
F
.51899 1.47228 m
.53165 1.47228 L
.53165 1.49602 L
.51899 1.49602 L
F
.27848 1.47228 m
.29114 1.47228 L
.29114 1.49602 L
.27848 1.49602 L
F
.29114 1.47228 m
.3038 1.47228 L
.3038 1.49602 L
.29114 1.49602 L
F
.3038 1.47228 m
.31646 1.47228 L
.31646 1.49602 L
.3038 1.49602 L
F
.31646 1.47228 m
.32911 1.47228 L
.32911 1.49602 L
.31646 1.49602 L
F
.34177 1.47228 m
.35443 1.47228 L
.35443 1.49602 L
.34177 1.49602 L
F
.46835 1.47228 m
.48101 1.47228 L
.48101 1.49602 L
.46835 1.49602 L
F
.48101 1.47228 m
.49367 1.47228 L
.49367 1.49602 L
.48101 1.49602 L
F
.49367 1.47228 m
.50633 1.47228 L
.50633 1.49602 L
.49367 1.49602 L
F
.03797 1.47228 m
.05063 1.47228 L
.05063 1.49602 L
.03797 1.49602 L
F
.06329 1.47228 m
.07595 1.47228 L
.07595 1.49602 L
.06329 1.49602 L
F
.55696 1.47228 m
.56962 1.47228 L
.56962 1.49602 L
.55696 1.49602 L
F
.4557 1.47228 m
.46835 1.47228 L
.46835 1.49602 L
.4557 1.49602 L
F
.5443 1.47228 m
.55696 1.47228 L
.55696 1.49602 L
.5443 1.49602 L
F
.44304 1.47228 m
.4557 1.47228 L
.4557 1.49602 L
.44304 1.49602 L
F
.43038 1.47228 m
.44304 1.47228 L
.44304 1.49602 L
.43038 1.49602 L
F
.82278 1.61476 m
.83544 1.61476 L
.83544 1.6385 L
.82278 1.6385 L
F
.87342 1.61476 m
.88608 1.61476 L
.88608 1.6385 L
.87342 1.6385 L
F
.81013 1.61476 m
.82278 1.61476 L
.82278 1.6385 L
.81013 1.6385 L
F
.86076 1.61476 m
.87342 1.61476 L
.87342 1.6385 L
.86076 1.6385 L
F
.07595 1.6385 m
.08861 1.6385 L
.08861 1.66225 L
.07595 1.66225 L
F
.8481 1.61476 m
.86076 1.61476 L
.86076 1.6385 L
.8481 1.6385 L
F
.94937 1.61476 m
.96203 1.61476 L
.96203 1.6385 L
.94937 1.6385 L
F
.92405 1.61476 m
.93671 1.61476 L
.93671 1.6385 L
.92405 1.6385 L
F
.89873 1.61476 m
.91139 1.61476 L
.91139 1.6385 L
.89873 1.6385 L
F
.83544 1.61476 m
.8481 1.61476 L
.8481 1.6385 L
.83544 1.6385 L
F
.93671 1.61476 m
.94937 1.61476 L
.94937 1.6385 L
.93671 1.6385 L
F
.96203 1.61476 m
.97468 1.61476 L
.97468 1.6385 L
.96203 1.6385 L
F
.97468 1.61476 m
.98734 1.61476 L
.98734 1.6385 L
.97468 1.6385 L
F
.98734 1.61476 m
1 1.61476 L
1 1.6385 L
.98734 1.6385 L
F
.03797 1.6385 m
.05063 1.6385 L
.05063 1.66225 L
.03797 1.66225 L
F
.02532 1.6385 m
.03797 1.6385 L
.03797 1.66225 L
.02532 1.66225 L
F
.06329 1.6385 m
.07595 1.6385 L
.07595 1.66225 L
.06329 1.66225 L
F
.11392 1.6385 m
.12658 1.6385 L
.12658 1.66225 L
.11392 1.66225 L
F
.05063 1.6385 m
.06329 1.6385 L
.06329 1.66225 L
.05063 1.66225 L
F
.10127 1.6385 m
.11392 1.6385 L
.11392 1.66225 L
.10127 1.66225 L
F
.08861 1.6385 m
.10127 1.6385 L
.10127 1.66225 L
.08861 1.66225 L
F
.63291 1.66225 m
.64557 1.66225 L
.64557 1.686 L
.63291 1.686 L
F
.64557 1.66225 m
.65823 1.66225 L
.65823 1.686 L
.64557 1.686 L
F
.65823 1.66225 m
.67089 1.66225 L
.67089 1.686 L
.65823 1.686 L
F
.67089 1.66225 m
.68354 1.66225 L
.68354 1.686 L
.67089 1.686 L
F
.6962 1.66225 m
.70886 1.66225 L
.70886 1.686 L
.6962 1.686 L
F
.40506 1.66225 m
.41772 1.66225 L
.41772 1.686 L
.40506 1.686 L
F
.4557 1.66225 m
.46835 1.66225 L
.46835 1.686 L
.4557 1.686 L
F
.39241 1.66225 m
.40506 1.66225 L
.40506 1.686 L
.39241 1.686 L
F
.44304 1.66225 m
.4557 1.66225 L
.4557 1.686 L
.44304 1.686 L
F
.43038 1.66225 m
.44304 1.66225 L
.44304 1.686 L
.43038 1.686 L
F
.58228 1.66225 m
.59494 1.66225 L
.59494 1.686 L
.58228 1.686 L
F
.59494 1.66225 m
.60759 1.66225 L
.60759 1.686 L
.59494 1.686 L
F
.62025 1.66225 m
.63291 1.66225 L
.63291 1.686 L
.62025 1.686 L
F
.88608 1.61476 m
.89873 1.61476 L
.89873 1.6385 L
.88608 1.6385 L
F
.91139 1.61476 m
.92405 1.61476 L
.92405 1.6385 L
.91139 1.6385 L
F
.56962 1.66225 m
.58228 1.66225 L
.58228 1.686 L
.56962 1.686 L
F
.5443 1.66225 m
.55696 1.66225 L
.55696 1.686 L
.5443 1.686 L
F
.68354 1.66225 m
.6962 1.66225 L
.6962 1.686 L
.68354 1.686 L
F
.60759 1.66225 m
.62025 1.66225 L
.62025 1.686 L
.60759 1.686 L
F
.55696 1.66225 m
.56962 1.66225 L
.56962 1.686 L
.55696 1.686 L
F
.6962 1.61476 m
.70886 1.61476 L
.70886 1.6385 L
.6962 1.6385 L
F
.5443 1.6385 m
.55696 1.6385 L
.55696 1.66225 L
.5443 1.66225 L
F
.68354 1.61476 m
.6962 1.61476 L
.6962 1.6385 L
.68354 1.6385 L
F
.53165 1.6385 m
.5443 1.6385 L
.5443 1.66225 L
.53165 1.66225 L
F
.58228 1.6385 m
.59494 1.6385 L
.59494 1.66225 L
.58228 1.66225 L
F
.51899 1.6385 m
.53165 1.6385 L
.53165 1.66225 L
.51899 1.66225 L
F
.64557 1.61476 m
.65823 1.61476 L
.65823 1.6385 L
.64557 1.6385 L
F
.65823 1.61476 m
.67089 1.61476 L
.67089 1.6385 L
.65823 1.6385 L
F
.67089 1.61476 m
.68354 1.61476 L
.68354 1.6385 L
.67089 1.6385 L
F
.72152 1.61476 m
.73418 1.61476 L
.73418 1.6385 L
.72152 1.6385 L
F
.70886 1.61476 m
.72152 1.61476 L
.72152 1.6385 L
.70886 1.6385 L
F
.48101 1.6385 m
.49367 1.6385 L
.49367 1.66225 L
.48101 1.66225 L
F
.49367 1.6385 m
.50633 1.6385 L
.50633 1.66225 L
.49367 1.66225 L
F
.50633 1.6385 m
.51899 1.6385 L
.51899 1.66225 L
.50633 1.66225 L
F
.68354 1.6385 m
.6962 1.6385 L
.6962 1.66225 L
.68354 1.66225 L
F
.70886 1.6385 m
.72152 1.6385 L
.72152 1.66225 L
.70886 1.66225 L
F
.56962 1.6385 m
.58228 1.6385 L
.58228 1.66225 L
.56962 1.66225 L
F
.46835 1.6385 m
.48101 1.6385 L
.48101 1.66225 L
.46835 1.66225 L
F
.55696 1.6385 m
.56962 1.6385 L
.56962 1.66225 L
.55696 1.66225 L
F
.4557 1.6385 m
.46835 1.6385 L
.46835 1.66225 L
.4557 1.66225 L
F
.44304 1.6385 m
.4557 1.6385 L
.4557 1.66225 L
.44304 1.66225 L
F
.60759 1.61476 m
.62025 1.61476 L
.62025 1.6385 L
.60759 1.6385 L
F
.62025 1.61476 m
.63291 1.61476 L
.63291 1.6385 L
.62025 1.6385 L
F
.63291 1.61476 m
.64557 1.61476 L
.64557 1.6385 L
.63291 1.6385 L
F
.49367 1.61476 m
.50633 1.61476 L
.50633 1.6385 L
.49367 1.6385 L
F
.51899 1.61476 m
.53165 1.61476 L
.53165 1.6385 L
.51899 1.6385 L
F
.01266 1.6385 m
.02532 1.6385 L
.02532 1.66225 L
.01266 1.66225 L
F
.59494 1.61476 m
.60759 1.61476 L
.60759 1.6385 L
.59494 1.6385 L
F
0 1.6385 m
.01266 1.6385 L
.01266 1.66225 L
0 1.66225 L
F
.58228 1.61476 m
.59494 1.61476 L
.59494 1.6385 L
.58228 1.6385 L
F
.56962 1.61476 m
.58228 1.61476 L
.58228 1.6385 L
.56962 1.6385 L
F
.74684 1.61476 m
.75949 1.61476 L
.75949 1.6385 L
.74684 1.6385 L
F
.79747 1.61476 m
.81013 1.61476 L
.81013 1.6385 L
.79747 1.6385 L
F
.73418 1.61476 m
.74684 1.61476 L
.74684 1.6385 L
.73418 1.6385 L
F
.78481 1.61476 m
.79747 1.61476 L
.79747 1.6385 L
.78481 1.6385 L
F
.77215 1.61476 m
.78481 1.61476 L
.78481 1.6385 L
.77215 1.6385 L
F
.55696 1.61476 m
.56962 1.61476 L
.56962 1.6385 L
.55696 1.6385 L
F
.53165 1.61476 m
.5443 1.61476 L
.5443 1.6385 L
.53165 1.6385 L
F
.50633 1.61476 m
.51899 1.61476 L
.51899 1.6385 L
.50633 1.6385 L
F
.75949 1.61476 m
.77215 1.61476 L
.77215 1.6385 L
.75949 1.6385 L
F
.5443 1.61476 m
.55696 1.61476 L
.55696 1.6385 L
.5443 1.6385 L
F
.1519 1.44853 m
.16456 1.44853 L
.16456 1.47228 L
.1519 1.47228 L
F
.12658 1.44853 m
.13924 1.44853 L
.13924 1.47228 L
.12658 1.47228 L
F
.18987 1.49602 m
.20253 1.49602 L
.20253 1.51977 L
.18987 1.51977 L
F
.10127 1.44853 m
.11392 1.44853 L
.11392 1.47228 L
.10127 1.47228 L
F
.03797 1.44853 m
.05063 1.44853 L
.05063 1.47228 L
.03797 1.47228 L
F
.01266 1.44853 m
.02532 1.44853 L
.02532 1.47228 L
.01266 1.47228 L
F
.13924 1.44853 m
.1519 1.44853 L
.1519 1.47228 L
.13924 1.47228 L
F
.08861 1.49602 m
.10127 1.49602 L
.10127 1.51977 L
.08861 1.51977 L
F
.11392 1.49602 m
.12658 1.49602 L
.12658 1.51977 L
.11392 1.51977 L
F
.11392 1.44853 m
.12658 1.44853 L
.12658 1.47228 L
.11392 1.47228 L
F
.08861 1.44853 m
.10127 1.44853 L
.10127 1.47228 L
.08861 1.47228 L
F
.21519 1.49602 m
.22785 1.49602 L
.22785 1.51977 L
.21519 1.51977 L
F
.24051 1.49602 m
.25316 1.49602 L
.25316 1.51977 L
.24051 1.51977 L
F
.22785 1.49602 m
.24051 1.49602 L
.24051 1.51977 L
.22785 1.51977 L
F
.3038 1.44853 m
.31646 1.44853 L
.31646 1.47228 L
.3038 1.47228 L
F
.18987 1.44853 m
.20253 1.44853 L
.20253 1.47228 L
.18987 1.47228 L
F
.31646 1.44853 m
.32911 1.44853 L
.32911 1.47228 L
.31646 1.47228 L
F
.29114 1.44853 m
.3038 1.44853 L
.3038 1.47228 L
.29114 1.47228 L
F
.17722 1.49602 m
.18987 1.49602 L
.18987 1.51977 L
.17722 1.51977 L
F
.20253 1.49602 m
.21519 1.49602 L
.21519 1.51977 L
.20253 1.51977 L
F
.16456 1.44853 m
.17722 1.44853 L
.17722 1.47228 L
.16456 1.47228 L
F
.05063 1.44853 m
.06329 1.44853 L
.06329 1.47228 L
.05063 1.47228 L
F
.06329 1.44853 m
.07595 1.44853 L
.07595 1.47228 L
.06329 1.47228 L
F
.02532 1.44853 m
.03797 1.44853 L
.03797 1.47228 L
.02532 1.47228 L
F
.07595 1.44853 m
.08861 1.44853 L
.08861 1.47228 L
.07595 1.47228 L
F
.25316 1.44853 m
.26582 1.44853 L
.26582 1.47228 L
.25316 1.47228 L
F
.26582 1.44853 m
.27848 1.44853 L
.27848 1.47228 L
.26582 1.47228 L
F
.27848 1.44853 m
.29114 1.44853 L
.29114 1.47228 L
.27848 1.47228 L
F
.92405 1.47228 m
.93671 1.47228 L
.93671 1.49602 L
.92405 1.49602 L
F
.89873 1.47228 m
.91139 1.47228 L
.91139 1.49602 L
.89873 1.49602 L
F
.58228 1.47228 m
.59494 1.47228 L
.59494 1.49602 L
.58228 1.49602 L
F
.60759 1.47228 m
.62025 1.47228 L
.62025 1.49602 L
.60759 1.49602 L
F
.79747 1.47228 m
.81013 1.47228 L
.81013 1.49602 L
.79747 1.49602 L
F
.78481 1.47228 m
.79747 1.47228 L
.79747 1.49602 L
.78481 1.49602 L
F
.91139 1.47228 m
.92405 1.47228 L
.92405 1.49602 L
.91139 1.49602 L
F
.63291 1.47228 m
.64557 1.47228 L
.64557 1.49602 L
.63291 1.49602 L
F
.62025 1.47228 m
.63291 1.47228 L
.63291 1.49602 L
.62025 1.49602 L
F
.86076 1.47228 m
.87342 1.47228 L
.87342 1.49602 L
.86076 1.49602 L
F
.88608 1.47228 m
.89873 1.47228 L
.89873 1.49602 L
.88608 1.49602 L
F
.64557 1.47228 m
.65823 1.47228 L
.65823 1.49602 L
.64557 1.49602 L
F
.59494 1.47228 m
.60759 1.47228 L
.60759 1.49602 L
.59494 1.49602 L
F
.87342 1.47228 m
.88608 1.47228 L
.88608 1.49602 L
.87342 1.49602 L
F
.12658 1.49602 m
.13924 1.49602 L
.13924 1.51977 L
.12658 1.51977 L
F
.16456 1.49602 m
.17722 1.49602 L
.17722 1.51977 L
.16456 1.51977 L
F
.68354 1.47228 m
.6962 1.47228 L
.6962 1.49602 L
.68354 1.49602 L
F
.13924 1.49602 m
.1519 1.49602 L
.1519 1.51977 L
.13924 1.51977 L
F
.10127 1.49602 m
.11392 1.49602 L
.11392 1.51977 L
.10127 1.51977 L
F
.82278 1.47228 m
.83544 1.47228 L
.83544 1.49602 L
.82278 1.49602 L
F
.1519 1.49602 m
.16456 1.49602 L
.16456 1.51977 L
.1519 1.51977 L
F
.83544 1.47228 m
.8481 1.47228 L
.8481 1.49602 L
.83544 1.49602 L
F
.8481 1.47228 m
.86076 1.47228 L
.86076 1.49602 L
.8481 1.49602 L
F
.6962 1.47228 m
.70886 1.47228 L
.70886 1.49602 L
.6962 1.49602 L
F
.81013 1.47228 m
.82278 1.47228 L
.82278 1.49602 L
.81013 1.49602 L
F
.65823 1.47228 m
.67089 1.47228 L
.67089 1.49602 L
.65823 1.49602 L
F
.67089 1.47228 m
.68354 1.47228 L
.68354 1.49602 L
.67089 1.49602 L
F
.73418 1.42478 m
.74684 1.42478 L
.74684 1.44853 L
.73418 1.44853 L
F
.74684 1.42478 m
.75949 1.42478 L
.75949 1.44853 L
.74684 1.44853 L
F
.75949 1.42478 m
.77215 1.42478 L
.77215 1.44853 L
.75949 1.44853 L
F
.01266 1.49602 m
.02532 1.49602 L
.02532 1.51977 L
.01266 1.51977 L
F
.93671 1.42478 m
.94937 1.42478 L
.94937 1.44853 L
.93671 1.44853 L
F
.16456 1.47228 m
.17722 1.47228 L
.17722 1.49602 L
.16456 1.49602 L
F
.03797 1.49602 m
.05063 1.49602 L
.05063 1.51977 L
.03797 1.51977 L
F
.96203 1.42478 m
.97468 1.42478 L
.97468 1.44853 L
.96203 1.44853 L
F
.82278 1.42478 m
.83544 1.42478 L
.83544 1.44853 L
.82278 1.44853 L
F
.72152 1.42478 m
.73418 1.42478 L
.73418 1.44853 L
.72152 1.44853 L
F
.83544 1.42478 m
.8481 1.42478 L
.8481 1.44853 L
.83544 1.44853 L
F
.81013 1.42478 m
.82278 1.42478 L
.82278 1.44853 L
.81013 1.44853 L
F
.6962 1.42478 m
.70886 1.42478 L
.70886 1.44853 L
.6962 1.44853 L
F
.70886 1.42478 m
.72152 1.42478 L
.72152 1.44853 L
.70886 1.44853 L
F
.02532 1.49602 m
.03797 1.49602 L
.03797 1.51977 L
.02532 1.51977 L
F
.1519 1.47228 m
.16456 1.47228 L
.16456 1.49602 L
.1519 1.49602 L
F
.88608 1.42478 m
.89873 1.42478 L
.89873 1.44853 L
.88608 1.44853 L
F
.91139 1.42478 m
.92405 1.42478 L
.92405 1.44853 L
.91139 1.44853 L
F
.92405 1.42478 m
.93671 1.42478 L
.93671 1.44853 L
.92405 1.44853 L
F
.74684 1.44853 m
.75949 1.44853 L
.75949 1.47228 L
.74684 1.47228 L
F
.72152 1.44853 m
.73418 1.44853 L
.73418 1.47228 L
.72152 1.47228 L
F
.89873 1.42478 m
.91139 1.42478 L
.91139 1.44853 L
.89873 1.44853 L
F
.97468 1.42478 m
.98734 1.42478 L
.98734 1.44853 L
.97468 1.44853 L
F
.98734 1.42478 m
1 1.42478 L
1 1.44853 L
.98734 1.44853 L
F
.94937 1.42478 m
.96203 1.42478 L
.96203 1.44853 L
.94937 1.44853 L
F
0 1.44853 m
.01266 1.44853 L
.01266 1.47228 L
0 1.47228 L
F
.8481 1.42478 m
.86076 1.42478 L
.86076 1.44853 L
.8481 1.44853 L
F
.86076 1.42478 m
.87342 1.42478 L
.87342 1.44853 L
.86076 1.44853 L
F
.87342 1.42478 m
.88608 1.42478 L
.88608 1.44853 L
.87342 1.44853 L
F
.20253 1.44853 m
.21519 1.44853 L
.21519 1.47228 L
.20253 1.47228 L
F
.93671 1.47228 m
.94937 1.47228 L
.94937 1.49602 L
.93671 1.49602 L
F
.24051 1.44853 m
.25316 1.44853 L
.25316 1.47228 L
.24051 1.47228 L
F
.21519 1.44853 m
.22785 1.44853 L
.22785 1.47228 L
.21519 1.47228 L
F
.06329 1.49602 m
.07595 1.49602 L
.07595 1.51977 L
.06329 1.51977 L
F
.96203 1.47228 m
.97468 1.47228 L
.97468 1.49602 L
.96203 1.49602 L
F
.97468 1.47228 m
.98734 1.47228 L
.98734 1.49602 L
.97468 1.49602 L
F
.17722 1.44853 m
.18987 1.44853 L
.18987 1.47228 L
.17722 1.47228 L
F
.98734 1.47228 m
1 1.47228 L
1 1.49602 L
.98734 1.49602 L
F
.22785 1.44853 m
.24051 1.44853 L
.24051 1.47228 L
.22785 1.47228 L
F
.94937 1.47228 m
.96203 1.47228 L
.96203 1.49602 L
.94937 1.49602 L
F
0 1.49602 m
.01266 1.49602 L
.01266 1.51977 L
0 1.51977 L
F
.05063 1.49602 m
.06329 1.49602 L
.06329 1.51977 L
.05063 1.51977 L
F
.78481 1.42478 m
.79747 1.42478 L
.79747 1.44853 L
.78481 1.44853 L
F
.77215 1.42478 m
.78481 1.42478 L
.78481 1.44853 L
.77215 1.44853 L
F
.07595 1.49602 m
.08861 1.49602 L
.08861 1.51977 L
.07595 1.51977 L
F
.79747 1.42478 m
.81013 1.42478 L
.81013 1.44853 L
.79747 1.44853 L
F
.6962 .3562 m
.70886 .3562 L
.70886 .36807 L
.70886 .37994 L
.6962 .37994 L
F
.34177 .45061 m
.34041 .45118 L
.32911 .45118 L
.32911 .42744 L
.34177 .42744 L
F
.72443 1.5079 m
.72152 1.5079 L
.72152 1.49602 L
.72785 1.49602 L
.72785 1.5036 L
F
.55078 1.37729 m
.55696 1.37729 L
.55696 1.40104 L
.5443 1.40104 L
.5443 1.38361 L
F
.71519 .37994 m
.72152 .37994 L
.72152 .40369 L
.70886 .40369 L
.70886 .37994 L
F
.48213 1.42478 m
.49367 1.42478 L
.49367 1.44853 L
.48101 1.44853 L
.48101 1.4252 L
F
.78131 .36807 m
.78481 .36807 L
.78481 .37994 L
.77848 .37994 L
.77848 .37141 L
F
.72152 .3393 m
.71976 .34432 L
.71519 .34432 L
.71519 .33245 L
.72152 .33245 L
F
.79747 .33245 m
.81013 .33245 L
.81013 .3562 L
.79747 .3562 L
.79747 .34432 L
F
.71984 .3562 m
.72152 .36128 L
.72152 .36807 L
.71519 .36807 L
.71519 .3562 L
F
.52483 1.40104 m
.53165 1.40104 L
.53165 1.42478 L
.51899 1.42478 L
.51899 1.40476 L
F
.72785 .37994 m
.73418 .37994 L
.73418 .40369 L
.72152 .40369 L
.72152 .37994 L
F
.77848 .37994 m
.78481 .37994 L
.78481 .40369 L
.77215 .40369 L
.77215 .37994 L
F
.77215 1.47228 m
.78481 1.47228 L
.78481 1.49602 L
.77848 1.49602 L
.77215 1.49602 L
F
.73418 .32003 m
.73341 .32058 L
.72785 .32058 L
.72785 .3087 L
.73418 .3087 L
F
.6962 .33245 m
.70886 .33245 L
.70886 .34432 L
.70886 .3562 L
.6962 .3562 L
F
.74684 .38254 m
.74684 .40369 L
.73418 .40369 L
.73418 .37994 L
.73855 .37994 L
F
.78131 1.5079 m
.77848 1.50456 L
.77848 1.49602 L
.78481 1.49602 L
.78481 1.5079 L
F
.72372 1.54352 m
.72785 1.54893 L
.72785 1.55539 L
.72152 1.55539 L
.72152 1.54352 L
F
.59666 .64115 m
.59494 .62067 L
.59494 .61741 L
.60759 .61741 L
.60759 .64115 L
F
.78481 1.54686 m
.78682 1.54352 L
.79114 1.54352 L
.79114 1.55539 L
.78481 1.55539 L
F
.73341 1.55539 m
.73418 1.55593 L
.73418 1.56726 L
.72785 1.56726 L
.72785 1.55539 L
F
.58639 1.30605 m
.59494 1.30605 L
.59494 1.3298 L
.58228 1.3298 L
.58228 1.31943 L
F
.17722 .61219 m
.17285 .61741 L
.16456 .61741 L
.16456 .59366 L
.17722 .59366 L
F
.16456 .62773 m
.15366 .64115 L
.1519 .64115 L
.1519 .61741 L
.16456 .61741 L
F
.13924 .66036 m
.13595 .6649 L
.12658 .6649 L
.12658 .64115 L
.13924 .64115 L
F
.12658 .67875 m
.12015 .68865 L
.11392 .68865 L
.11392 .6649 L
.12658 .6649 L
F
.07284 1.06859 m
.07595 1.08224 L
.07595 1.09233 L
.06329 1.09233 L
.06329 1.06859 L
F
.6962 1.54352 m
.70886 1.54352 L
.70886 1.55539 L
.70886 1.56726 L
.6962 1.56726 L
F
.72152 1.51469 m
.71984 1.51977 L
.71519 1.51977 L
.71519 1.5079 L
.72152 1.5079 L
F
.08576 1.11608 m
.08861 1.12365 L
.08861 1.13983 L
.07595 1.13983 L
.07595 1.11608 L
F
.09509 1.13983 m
.10127 1.15305 L
.10127 1.16357 L
.08861 1.16357 L
.08861 1.13983 L
F
.77215 1.55956 m
.77845 1.55539 L
.77848 1.55539 L
.77848 1.56726 L
.77215 1.56726 L
F
.78682 .33245 m
.78481 .32911 L
.78481 .32058 L
.79114 .32058 L
.79114 .33245 L
F
.56771 1.35355 m
.56962 1.35355 L
.56962 1.37729 L
.55696 1.37729 L
.55696 1.36897 L
F
.79747 1.54352 m
.81013 1.54352 L
.81013 1.56726 L
.79747 1.56726 L
.79747 1.55539 L
F
.71976 1.53164 m
.72152 1.53667 L
.72152 1.54352 L
.71519 1.54352 L
.71519 1.53164 L
F
.48213 .45118 m
.48101 .45077 L
.48101 .42744 L
.49367 .42744 L
.49367 .45118 L
F
.25316 .52048 m
.25129 .52242 L
.24051 .52242 L
.24051 .49867 L
.25316 .49867 L
F
.3038 .47264 m
.30044 .47493 L
.29114 .47493 L
.29114 .45118 L
.3038 .45118 L
F
.56771 .52242 m
.55696 .507 L
.55696 .49867 L
.56962 .49867 L
.56962 .52242 L
F
.24051 .53579 m
.23138 .54617 L
.22785 .54617 L
.22785 .52242 L
.24051 .52242 L
F
.21519 .5662 m
.212 .56991 L
.20253 .56991 L
.20253 .54617 L
.21519 .54617 L
F
.57892 .54617 m
.56962 .52629 L
.56962 .52242 L
.58228 .52242 L
.58228 .54617 L
F
.20253 .58182 m
.19243 .59366 L
.18987 .59366 L
.18987 .56991 L
.20253 .56991 L
F
.55078 .49867 m
.5443 .49236 L
.5443 .47493 L
.55696 .47493 L
.55696 .49867 L
F
.58639 .56991 m
.58228 .55654 L
.58228 .54617 L
.59494 .54617 L
.59494 .56991 L
F
.10127 .72292 m
.09509 .73614 L
.08861 .73614 L
.08861 .71239 L
.10127 .71239 L
F
.77845 .32058 m
.77215 .31641 L
.77215 .3087 L
.77848 .3087 L
.77848 .32058 L
F
.11392 .69915 m
.10649 .71239 L
.10127 .71239 L
.10127 .68865 L
.11392 .68865 L
F
.07595 .79373 m
.07284 .80738 L
.06329 .80738 L
.06329 .78363 L
.07595 .78363 L
F
.06329 .88914 m
.0627 .90236 L
.05063 .90236 L
.05063 .87862 L
.06329 .87862 L
F
.72785 .32703 m
.72372 .33245 L
.72152 .33245 L
.72152 .32058 L
.72785 .32058 L
F
.52483 .47493 m
.51899 .4712 L
.51899 .45118 L
.53165 .45118 L
.53165 .47493 L
F
.08861 .75232 m
.08576 .75989 L
.07595 .75989 L
.07595 .73614 L
.08861 .73614 L
F
.27848 .49417 m
.27338 .49867 L
.26582 .49867 L
.26582 .47493 L
.27848 .47493 L
F
.17285 1.25856 m
.17722 1.26378 L
.17722 1.28231 L
.16456 1.28231 L
.16456 1.25856 L
F
.71519 1.56726 m
.72152 1.56726 L
.72152 1.59101 L
.70886 1.59101 L
.70886 1.56726 L
F
.15366 1.23481 m
.16456 1.24824 L
.16456 1.25856 L
.1519 1.25856 L
.1519 1.23481 L
F
.72785 1.56726 m
.73418 1.56726 L
.73418 1.59101 L
.72152 1.59101 L
.72152 1.56726 L
F
.74051 1.56726 m
.74684 1.56726 L
.74684 1.59101 L
.73418 1.59101 L
.73418 1.56726 L
F
.75316 1.56726 m
.75949 1.56726 L
.75949 1.59101 L
.74684 1.59101 L
.74684 1.56726 L
F
.74684 .28496 m
.75949 .28496 L
.75949 .3087 L
.75316 .3087 L
.74684 .3087 L
F
.75949 .28496 m
.77215 .28496 L
.77215 .3087 L
.76582 .3087 L
.75949 .3087 L
F
.77215 .28496 m
.78481 .28496 L
.78481 .3087 L
.77848 .3087 L
.77215 .3087 L
F
.212 1.30605 m
.21519 1.30976 L
.21519 1.3298 L
.20253 1.3298 L
.20253 1.30605 L
F
.79747 .3087 m
.81013 .3087 L
.81013 .33245 L
.79747 .33245 L
.79747 .32058 L
F
.57892 1.3298 m
.58228 1.3298 L
.58228 1.35355 L
.56962 1.35355 L
.56962 1.34967 L
F
.6962 1.51977 m
.70886 1.51977 L
.70886 1.53164 L
.70886 1.54352 L
.6962 1.54352 L
F
.77848 1.56726 m
.78481 1.56726 L
.78481 1.59101 L
.77215 1.59101 L
.77215 1.56726 L
F
.59666 1.23481 m
.60759 1.23481 L
.60759 1.25856 L
.59494 1.25856 L
.59494 1.2553 L
F
.70886 1.47228 m
.72152 1.47228 L
.72152 1.49602 L
.71519 1.49602 L
.70886 1.49602 L
F
.72152 1.47228 m
.73418 1.47228 L
.73418 1.49602 L
.72785 1.49602 L
.72152 1.49602 L
F
.72785 .37236 m
.72785 .37994 L
.72152 .37994 L
.72152 .36807 L
.72443 .36807 L
F
.79747 1.51977 m
.81013 1.51977 L
.81013 1.54352 L
.79747 1.54352 L
.79747 1.53164 L
F
.76582 1.56726 m
.77215 1.56726 L
.77215 1.59101 L
.75949 1.59101 L
.75949 1.56726 L
F
.79114 1.56726 m
.79747 1.56726 L
.79747 1.59101 L
.78481 1.59101 L
.78481 1.56726 L
F
.30044 1.40104 m
.3038 1.40333 L
.3038 1.42478 L
.29114 1.42478 L
.29114 1.40104 L
F
.34041 1.42478 m
.34177 1.42535 L
.34177 1.44853 L
.32911 1.44853 L
.32911 1.42478 L
F
.74684 1.49343 m
.73855 1.49602 L
.73418 1.49602 L
.73418 1.47228 L
.74684 1.47228 L
F
.78481 .28496 m
.79747 .28496 L
.79747 .3087 L
.79114 .3087 L
.78481 .3087 L
F
.13595 1.21107 m
.13924 1.2156 L
.13924 1.23481 L
.12658 1.23481 L
.12658 1.21107 L
F
.27338 1.37729 m
.27848 1.38179 L
.27848 1.40104 L
.26582 1.40104 L
.26582 1.37729 L
F
.23138 1.3298 m
.24051 1.34018 L
.24051 1.35355 L
.22785 1.35355 L
.22785 1.3298 L
F
.0627 .9736 m
.06329 .98683 L
.06329 .99735 L
.05063 .99735 L
.05063 .9736 L
F
.25129 1.35355 m
.25316 1.35549 L
.25316 1.37729 L
.24051 1.37729 L
.24051 1.35355 L
F
.6962 1.49602 m
.70886 1.49602 L
.70886 1.5079 L
.70886 1.51977 L
.6962 1.51977 L
F
.70886 .28496 m
.72152 .28496 L
.72152 .3087 L
.71519 .3087 L
.70886 .3087 L
F
.73418 .28496 m
.74684 .28496 L
.74684 .3087 L
.74051 .3087 L
.73418 .3087 L
F
.6962 .3087 m
.70886 .3087 L
.70886 .32058 L
.70886 .33245 L
.6962 .33245 L
F
.72152 .28496 m
.73418 .28496 L
.73418 .3087 L
.72785 .3087 L
.72152 .3087 L
F
.12015 1.18732 m
.12658 1.19722 L
.12658 1.21107 L
.11392 1.21107 L
.11392 1.18732 L
F
.19243 1.28231 m
.20253 1.29415 L
.20253 1.30605 L
.18987 1.30605 L
.18987 1.28231 L
F
.10649 1.16357 m
.11392 1.17681 L
.11392 1.18732 L
.10127 1.18732 L
.10127 1.16357 L
F
.75949 .38202 m
.76546 .37994 L
.76582 .37994 L
.77215 .37994 L
.77215 .40369 L
.75949 .40369 L
F
.76546 1.49602 m
.75949 1.49394 L
.75949 1.47228 L
.77215 1.47228 L
.77215 1.49602 L
.76582 1.49602 L
F
.78771 1.51977 m
.78481 1.51337 L
.78481 1.5079 L
.78481 1.49602 L
.79747 1.49602 L
.79747 1.51977 L
.79114 1.51977 L
F
.78481 .3626 m
.78771 .3562 L
.79114 .3562 L
.79747 .3562 L
.79747 .37994 L
.78481 .37994 L
.78481 .36807 L
F
0 g
.5 Mabswid
.73418 .32003 m
.73341 .32058 L
.72785 .32703 L
.72372 .33245 L
.72152 .3393 L
.71976 .34432 L
.71984 .3562 L
.72152 .36128 L
.72443 .36807 L
.72785 .37236 L
.73418 .37773 L
.73855 .37994 L
.74684 .38254 L
.75949 .38202 L
.76546 .37994 L
.76582 .37984 L
.77215 .3766 L
.77848 .37141 L
.78131 .36807 L
.78481 .3626 L
.78771 .3562 L
.7894 .34432 L
.78682 .33245 L
.78481 .32911 L
.77848 .32061 L
.77845 .32058 L
.77215 .31641 L
.76582 .31374 L
.75949 .31248 L
.75316 .3125 L
.74684 .31377 L
.74051 .31629 L
.73418 .32003 L
s
.34177 .45061 m
.34041 .45118 L
.32911 .45709 L
.31646 .46457 L
.3038 .47264 L
.30044 .47493 L
.29114 .48305 L
.27848 .49417 L
.27338 .49867 L
.26582 .5071 L
.25316 .52048 L
.25129 .52242 L
.24051 .53579 L
.23138 .54617 L
.22785 .55076 L
.21519 .5662 L
.212 .56991 L
.20253 .58182 L
.19243 .59366 L
.18987 .59681 L
.17722 .61219 L
.17285 .61741 L
.16456 .62773 L
.15366 .64115 L
.1519 .64342 L
.13924 .66036 L
.13595 .6649 L
.12658 .67875 L
.12015 .68865 L
.11392 .69915 L
.10649 .71239 L
.10127 .72292 L
.09509 .73614 L
.08861 .75232 L
.08576 .75989 L
.07841 .78363 L
.07595 .79373 L
.07284 .80738 L
.06876 .83112 L
.06581 .85487 L
.06382 .87862 L
.06329 .88914 L
.0627 .90236 L
.06221 .92611 L
.06221 .94986 L
.0627 .9736 L
.06329 .98683 L
.06382 .99735 L
.06581 1.0211 L
.06876 1.04484 L
Mistroke
.07284 1.06859 L
.07595 1.08224 L
.07841 1.09233 L
.08576 1.11608 L
.08861 1.12365 L
.09509 1.13983 L
.10127 1.15305 L
.10649 1.16357 L
.11392 1.17681 L
.12015 1.18732 L
.12658 1.19722 L
.13595 1.21107 L
.13924 1.2156 L
.1519 1.23254 L
.15366 1.23481 L
.16456 1.24824 L
.17285 1.25856 L
.17722 1.26378 L
.18987 1.27916 L
.19243 1.28231 L
.20253 1.29415 L
.212 1.30605 L
.21519 1.30976 L
.22785 1.32521 L
.23138 1.3298 L
.24051 1.34018 L
.25129 1.35355 L
.25316 1.35549 L
.26582 1.36887 L
.27338 1.37729 L
.27848 1.38179 L
.29114 1.39292 L
.30044 1.40104 L
.3038 1.40333 L
.31646 1.4114 L
.32911 1.41887 L
.34041 1.42478 L
.34177 1.42535 L
.35443 1.42982 L
.36709 1.43344 L
.37975 1.43615 L
.39241 1.43791 L
.40506 1.43872 L
.41772 1.4386 L
.43038 1.43756 L
.44304 1.43565 L
.4557 1.43291 L
.46835 1.42941 L
.48101 1.4252 L
.48213 1.42478 L
Mistroke
.49367 1.41919 L
.50633 1.41233 L
.51899 1.40476 L
.52483 1.40104 L
.53165 1.39518 L
.5443 1.38361 L
.55078 1.37729 L
.55696 1.36897 L
.56771 1.35355 L
.56962 1.34967 L
.57892 1.3298 L
.58228 1.31943 L
.58639 1.30605 L
.59135 1.28231 L
.59465 1.25856 L
.59494 1.2553 L
.59666 1.23481 L
.59786 1.21107 L
.59854 1.18732 L
.59888 1.16357 L
.59904 1.13983 L
.5991 1.11608 L
.59911 1.09233 L
.59911 1.06859 L
.59911 1.04484 L
.59911 1.0211 L
.59911 .99735 L
.59911 .9736 L
.59911 .94986 L
.59911 .92611 L
.59911 .90236 L
.59911 .87862 L
.59911 .85487 L
.59911 .83112 L
.59911 .80738 L
.59911 .78363 L
.5991 .75989 L
.59904 .73614 L
.59888 .71239 L
.59854 .68865 L
.59786 .6649 L
.59666 .64115 L
.59494 .62067 L
.59465 .61741 L
.59135 .59366 L
.58639 .56991 L
.58228 .55654 L
.57892 .54617 L
.56962 .52629 L
.56771 .52242 L
Mistroke
.55696 .507 L
.55078 .49867 L
.5443 .49236 L
.53165 .48079 L
.52483 .47493 L
.51899 .4712 L
.50633 .46364 L
.49367 .45678 L
.48213 .45118 L
.48101 .45077 L
.46835 .44656 L
.4557 .44306 L
.44304 .44032 L
.43038 .43841 L
.41772 .43737 L
.40506 .43724 L
.39241 .43806 L
.37975 .43982 L
.36709 .44252 L
.35443 .44614 L
.34177 .45061 L
Mfstroke
.74684 1.49343 m
.73855 1.49602 L
.73418 1.49824 L
.72785 1.5036 L
.72443 1.5079 L
.72152 1.51469 L
.71984 1.51977 L
.71976 1.53164 L
.72152 1.53667 L
.72372 1.54352 L
.72785 1.54893 L
.73341 1.55539 L
.73418 1.55593 L
.74051 1.55968 L
.74684 1.5622 L
.75316 1.56347 L
.75949 1.56349 L
.76582 1.56223 L
.77215 1.55956 L
.77845 1.55539 L
.77848 1.55536 L
.78481 1.54686 L
.78682 1.54352 L
.7894 1.53164 L
.78771 1.51977 L
.78481 1.51337 L
.78131 1.5079 L
.77848 1.50456 L
.77215 1.49937 L
.76582 1.49613 L
.76546 1.49602 L
.75949 1.49394 L
.74684 1.49343 L
s
1 g
.59314 .86824 m
.59571 .86824 L
.59571 .87305 L
.59314 .87305 L
F
.54699 .87305 m
.54955 .87305 L
.54955 .87786 L
.54699 .87786 L
F
.5393 .87305 m
.54186 .87305 L
.54186 .87786 L
.5393 .87786 L
F
.59058 .86824 m
.59314 .86824 L
.59314 .87305 L
.59058 .87305 L
F
.54443 .87305 m
.54699 .87305 L
.54699 .87786 L
.54443 .87786 L
F
.5752 .86824 m
.57776 .86824 L
.57776 .87305 L
.5752 .87305 L
F
.54955 .87305 m
.55212 .87305 L
.55212 .87786 L
.54955 .87786 L
F
.54186 .87305 m
.54443 .87305 L
.54443 .87786 L
.54186 .87786 L
F
.59058 .87786 m
.59314 .87786 L
.59314 .88267 L
.59058 .88267 L
F
.59314 .87786 m
.59571 .87786 L
.59571 .88267 L
.59314 .88267 L
F
.5752 .87786 m
.57776 .87786 L
.57776 .88267 L
.5752 .88267 L
F
.57776 .87786 m
.58032 .87786 L
.58032 .88267 L
.57776 .88267 L
F
.58802 .87786 m
.59058 .87786 L
.59058 .88267 L
.58802 .88267 L
F
.59571 .86824 m
.59827 .86824 L
.59827 .87305 L
.59571 .87305 L
F
.58545 .87786 m
.58802 .87786 L
.58802 .88267 L
.58545 .88267 L
F
.57263 .87305 m
.5752 .87305 L
.5752 .87786 L
.57263 .87786 L
F
.5752 .87305 m
.57776 .87305 L
.57776 .87786 L
.5752 .87786 L
F
.57776 .87305 m
.58032 .87305 L
.58032 .87786 L
.57776 .87786 L
F
.58032 .87305 m
.58289 .87305 L
.58289 .87786 L
.58032 .87786 L
F
.58289 .87305 m
.58545 .87305 L
.58545 .87786 L
.58289 .87786 L
F
.58545 .87305 m
.58802 .87305 L
.58802 .87786 L
.58545 .87786 L
F
.58802 .87305 m
.59058 .87305 L
.59058 .87786 L
.58802 .87786 L
F
.59058 .87305 m
.59314 .87305 L
.59314 .87786 L
.59058 .87786 L
F
.57263 .86824 m
.5752 .86824 L
.5752 .87305 L
.57263 .87305 L
F
.58545 .86824 m
.58802 .86824 L
.58802 .87305 L
.58545 .87305 L
F
.57776 .86824 m
.58032 .86824 L
.58032 .87305 L
.57776 .87305 L
F
.57007 .86824 m
.57263 .86824 L
.57263 .87305 L
.57007 .87305 L
F
.58032 .86824 m
.58289 .86824 L
.58289 .87305 L
.58032 .87305 L
F
.58289 .86824 m
.58545 .86824 L
.58545 .87305 L
.58289 .87305 L
F
.58802 .86824 m
.59058 .86824 L
.59058 .87305 L
.58802 .87305 L
F
.56494 .87786 m
.5675 .87786 L
.5675 .88267 L
.56494 .88267 L
F
.5675 .87786 m
.57007 .87786 L
.57007 .88267 L
.5675 .88267 L
F
.57007 .87786 m
.57263 .87786 L
.57263 .88267 L
.57007 .88267 L
F
.57263 .87786 m
.5752 .87786 L
.5752 .88267 L
.57263 .88267 L
F
.59314 .87305 m
.59571 .87305 L
.59571 .87786 L
.59314 .87786 L
F
.59571 .87305 m
.59827 .87305 L
.59827 .87786 L
.59571 .87786 L
F
.5393 .87786 m
.54186 .87786 L
.54186 .88267 L
.5393 .88267 L
F
.54186 .87786 m
.54443 .87786 L
.54443 .88267 L
.54186 .88267 L
F
.54955 .99811 m
.55212 .99811 L
.55212 1.00292 L
.54955 1.00292 L
F
.55981 .87786 m
.56238 .87786 L
.56238 .88267 L
.55981 .88267 L
F
.55468 .99811 m
.55725 .99811 L
.55725 1.00292 L
.55468 1.00292 L
F
.54699 .99811 m
.54955 .99811 L
.54955 1.00292 L
.54699 1.00292 L
F
.55468 .87786 m
.55725 .87786 L
.55725 .88267 L
.55468 .88267 L
F
.55725 .87786 m
.55981 .87786 L
.55981 .88267 L
.55725 .88267 L
F
.56238 .87786 m
.56494 .87786 L
.56494 .88267 L
.56238 .88267 L
F
.55468 .88267 m
.55725 .88267 L
.55725 .88748 L
.55468 .88748 L
F
.54699 .88267 m
.54955 .88267 L
.54955 .88748 L
.54699 .88748 L
F
.5393 .88267 m
.54186 .88267 L
.54186 .88748 L
.5393 .88748 L
F
.55212 .88267 m
.55468 .88267 L
.55468 .88748 L
.55212 .88748 L
F
.58032 .87786 m
.58289 .87786 L
.58289 .88267 L
.58032 .88267 L
F
.58289 .87786 m
.58545 .87786 L
.58545 .88267 L
.58289 .88267 L
F
.54955 .88267 m
.55212 .88267 L
.55212 .88748 L
.54955 .88748 L
F
.54443 .87786 m
.54699 .87786 L
.54699 .88267 L
.54443 .88267 L
F
.54699 .87786 m
.54955 .87786 L
.54955 .88267 L
.54699 .88267 L
F
.54955 .87786 m
.55212 .87786 L
.55212 .88267 L
.54955 .88267 L
F
.55212 .87786 m
.55468 .87786 L
.55468 .88267 L
.55212 .88267 L
F
.54443 .88267 m
.54699 .88267 L
.54699 .88748 L
.54443 .88748 L
F
.59571 .87786 m
.59827 .87786 L
.59827 .88267 L
.59571 .88267 L
F
.54186 .88267 m
.54443 .88267 L
.54443 .88748 L
.54186 .88748 L
F
.57007 .89229 m
.57263 .89229 L
.57263 .8971 L
.57007 .8971 L
F
.57263 .89229 m
.5752 .89229 L
.5752 .8971 L
.57263 .8971 L
F
.5752 .89229 m
.57776 .89229 L
.57776 .8971 L
.5752 .8971 L
F
.57776 .89229 m
.58032 .89229 L
.58032 .8971 L
.57776 .8971 L
F
.57776 .88267 m
.58032 .88267 L
.58032 .88748 L
.57776 .88748 L
F
.58032 .88267 m
.58289 .88267 L
.58289 .88748 L
.58032 .88748 L
F
.58289 .88267 m
.58545 .88267 L
.58545 .88748 L
.58289 .88748 L
F
.58545 .88267 m
.58802 .88267 L
.58802 .88748 L
.58545 .88748 L
F
.59314 .89229 m
.59571 .89229 L
.59571 .8971 L
.59314 .8971 L
F
.56494 .89229 m
.5675 .89229 L
.5675 .8971 L
.56494 .8971 L
F
.5393 .8971 m
.54186 .8971 L
.54186 .90191 L
.5393 .90191 L
F
.59058 .89229 m
.59314 .89229 L
.59314 .8971 L
.59058 .8971 L
F
.55981 .89229 m
.56238 .89229 L
.56238 .8971 L
.55981 .8971 L
F
.56238 .89229 m
.56494 .89229 L
.56494 .8971 L
.56238 .8971 L
F
.5675 .89229 m
.57007 .89229 L
.57007 .8971 L
.5675 .8971 L
F
.5752 .88267 m
.57776 .88267 L
.57776 .88748 L
.5752 .88748 L
F
.5675 .88267 m
.57007 .88267 L
.57007 .88748 L
.5675 .88748 L
F
.55981 .88267 m
.56238 .88267 L
.56238 .88748 L
.55981 .88748 L
F
.57263 .88267 m
.5752 .88267 L
.5752 .88748 L
.57263 .88748 L
F
.56238 .88748 m
.56494 .88748 L
.56494 .89229 L
.56238 .89229 L
F
.56494 .88748 m
.5675 .88748 L
.5675 .89229 L
.56494 .89229 L
F
.57007 .88267 m
.57263 .88267 L
.57263 .88748 L
.57007 .88748 L
F
.58802 .88267 m
.59058 .88267 L
.59058 .88748 L
.58802 .88748 L
F
.59058 .88267 m
.59314 .88267 L
.59314 .88748 L
.59058 .88748 L
F
.59314 .88267 m
.59571 .88267 L
.59571 .88748 L
.59314 .88748 L
F
.59571 .88267 m
.59827 .88267 L
.59827 .88748 L
.59571 .88748 L
F
.56494 .88267 m
.5675 .88267 L
.5675 .88748 L
.56494 .88748 L
F
.55725 .88267 m
.55981 .88267 L
.55981 .88748 L
.55725 .88748 L
F
.56238 .88267 m
.56494 .88267 L
.56494 .88748 L
.56238 .88748 L
F
.54699 .89229 m
.54955 .89229 L
.54955 .8971 L
.54699 .8971 L
F
.5393 .89229 m
.54186 .89229 L
.54186 .8971 L
.5393 .8971 L
F
.56494 .87305 m
.5675 .87305 L
.5675 .87786 L
.56494 .87786 L
F
.54443 .89229 m
.54699 .89229 L
.54699 .8971 L
.54443 .8971 L
F
.55725 .89229 m
.55981 .89229 L
.55981 .8971 L
.55725 .8971 L
F
.54955 .89229 m
.55212 .89229 L
.55212 .8971 L
.54955 .8971 L
F
.54186 .89229 m
.54443 .89229 L
.54443 .8971 L
.54186 .8971 L
F
.55468 .89229 m
.55725 .89229 L
.55725 .8971 L
.55468 .8971 L
F
.55212 .87305 m
.55468 .87305 L
.55468 .87786 L
.55212 .87786 L
F
.55468 .87305 m
.55725 .87305 L
.55725 .87786 L
.55468 .87786 L
F
.55725 .87305 m
.55981 .87305 L
.55981 .87786 L
.55725 .87786 L
F
.55981 .87305 m
.56238 .87305 L
.56238 .87786 L
.55981 .87786 L
F
.57007 .87305 m
.57263 .87305 L
.57263 .87786 L
.57007 .87786 L
F
.56238 .87305 m
.56494 .87305 L
.56494 .87786 L
.56238 .87786 L
F
.5675 .87305 m
.57007 .87305 L
.57007 .87786 L
.5675 .87786 L
F
.58545 .89229 m
.58802 .89229 L
.58802 .8971 L
.58545 .8971 L
F
.58802 .89229 m
.59058 .89229 L
.59058 .8971 L
.58802 .8971 L
F
.58802 .88748 m
.59058 .88748 L
.59058 .89229 L
.58802 .89229 L
F
.59058 .88748 m
.59314 .88748 L
.59314 .89229 L
.59058 .89229 L
F
.58289 .89229 m
.58545 .89229 L
.58545 .8971 L
.58289 .8971 L
F
.59571 .89229 m
.59827 .89229 L
.59827 .8971 L
.59571 .8971 L
F
.58032 .89229 m
.58289 .89229 L
.58289 .8971 L
.58032 .8971 L
F
.58545 .88748 m
.58802 .88748 L
.58802 .89229 L
.58545 .89229 L
F
.57776 .88748 m
.58032 .88748 L
.58032 .89229 L
.57776 .89229 L
F
.55212 .89229 m
.55468 .89229 L
.55468 .8971 L
.55212 .8971 L
F
.58289 .88748 m
.58545 .88748 L
.58545 .89229 L
.58289 .89229 L
F
.59314 .88748 m
.59571 .88748 L
.59571 .89229 L
.59314 .89229 L
F
.59571 .88748 m
.59827 .88748 L
.59827 .89229 L
.59571 .89229 L
F
.58032 .88748 m
.58289 .88748 L
.58289 .89229 L
.58032 .89229 L
F
.59058 .97887 m
.59314 .97887 L
.59314 .98368 L
.59058 .98368 L
F
.58802 .97887 m
.59058 .97887 L
.59058 .98368 L
.58802 .98368 L
F
.55981 .98368 m
.56238 .98368 L
.56238 .98849 L
.55981 .98849 L
F
.56238 .98368 m
.56494 .98368 L
.56494 .98849 L
.56238 .98849 L
F
.58545 .97887 m
.58802 .97887 L
.58802 .98368 L
.58545 .98368 L
F
.54186 .98368 m
.54443 .98368 L
.54443 .98849 L
.54186 .98849 L
F
.59314 .97887 m
.59571 .97887 L
.59571 .98368 L
.59314 .98368 L
F
.58289 .97887 m
.58545 .97887 L
.58545 .98368 L
.58289 .98368 L
F
.55725 .98368 m
.55981 .98368 L
.55981 .98849 L
.55725 .98849 L
F
.54699 .98368 m
.54955 .98368 L
.54955 .98849 L
.54699 .98849 L
F
.55468 .98368 m
.55725 .98368 L
.55725 .98849 L
.55468 .98849 L
F
.55212 .98368 m
.55468 .98368 L
.55468 .98849 L
.55212 .98849 L
F
.56494 .98368 m
.5675 .98368 L
.5675 .98849 L
.56494 .98849 L
F
.5675 .98368 m
.57007 .98368 L
.57007 .98849 L
.5675 .98849 L
F
.54955 .98368 m
.55212 .98368 L
.55212 .98849 L
.54955 .98849 L
F
.56238 1.00292 m
.56494 1.00292 L
.56494 1.00773 L
.56238 1.00773 L
F
.5675 1.00292 m
.57007 1.00292 L
.57007 1.00773 L
.5675 1.00773 L
F
.54699 1.00292 m
.54955 1.00292 L
.54955 1.00773 L
.54699 1.00773 L
F
.56494 1.00292 m
.5675 1.00292 L
.5675 1.00773 L
.56494 1.00773 L
F
.58802 .99811 m
.59058 .99811 L
.59058 1.00292 L
.58802 1.00292 L
F
.58545 .99811 m
.58802 .99811 L
.58802 1.00292 L
.58545 1.00292 L
F
.55725 1.00292 m
.55981 1.00292 L
.55981 1.00773 L
.55725 1.00773 L
F
.55981 1.00292 m
.56238 1.00292 L
.56238 1.00773 L
.55981 1.00773 L
F
.5393 .98368 m
.54186 .98368 L
.54186 .98849 L
.5393 .98849 L
F
.55212 1.00292 m
.55468 1.00292 L
.55468 1.00773 L
.55212 1.00773 L
F
.54443 .98368 m
.54699 .98368 L
.54699 .98849 L
.54443 .98849 L
F
.59571 .97887 m
.59827 .97887 L
.59827 .98368 L
.59571 .98368 L
F
.55468 1.00292 m
.55725 1.00292 L
.55725 1.00773 L
.55468 1.00773 L
F
.54443 1.00292 m
.54699 1.00292 L
.54699 1.00773 L
.54443 1.00773 L
F
.54955 1.00292 m
.55212 1.00292 L
.55212 1.00773 L
.54955 1.00773 L
F
.5752 .98849 m
.57776 .98849 L
.57776 .9933 L
.5752 .9933 L
F
.56494 .97887 m
.5675 .97887 L
.5675 .98368 L
.56494 .98368 L
F
.58032 .98849 m
.58289 .98849 L
.58289 .9933 L
.58032 .9933 L
F
.57263 .98849 m
.5752 .98849 L
.5752 .9933 L
.57263 .9933 L
F
.55725 .97887 m
.55981 .97887 L
.55981 .98368 L
.55725 .98368 L
F
.55981 .97887 m
.56238 .97887 L
.56238 .98368 L
.55981 .98368 L
F
.56238 .97887 m
.56494 .97887 L
.56494 .98368 L
.56238 .98368 L
F
.5675 .97887 m
.57007 .97887 L
.57007 .98368 L
.5675 .98368 L
F
.56494 .98849 m
.5675 .98849 L
.5675 .9933 L
.56494 .9933 L
F
.57007 .98849 m
.57263 .98849 L
.57263 .9933 L
.57007 .9933 L
F
.5393 .9933 m
.54186 .9933 L
.54186 .99811 L
.5393 .99811 L
F
.5675 .98849 m
.57007 .98849 L
.57007 .9933 L
.5675 .9933 L
F
.56238 .98849 m
.56494 .98849 L
.56494 .9933 L
.56238 .9933 L
F
.57776 .98849 m
.58032 .98849 L
.58032 .9933 L
.57776 .9933 L
F
.55981 .98849 m
.56238 .98849 L
.56238 .9933 L
.55981 .9933 L
F
.59571 .97406 m
.59827 .97406 L
.59827 .97887 L
.59571 .97887 L
F
.55212 .97887 m
.55468 .97887 L
.55468 .98368 L
.55212 .98368 L
F
.54443 .97887 m
.54699 .97887 L
.54699 .98368 L
.54443 .98368 L
F
.59314 .97406 m
.59571 .97406 L
.59571 .97887 L
.59314 .97887 L
F
.54699 .97887 m
.54955 .97887 L
.54955 .98368 L
.54699 .98368 L
F
.54955 .97887 m
.55212 .97887 L
.55212 .98368 L
.54955 .98368 L
F
.55468 .97887 m
.55725 .97887 L
.55725 .98368 L
.55468 .98368 L
F
.58032 .97887 m
.58289 .97887 L
.58289 .98368 L
.58032 .98368 L
F
.57007 .97887 m
.57263 .97887 L
.57263 .98368 L
.57007 .98368 L
F
.57776 .97887 m
.58032 .97887 L
.58032 .98368 L
.57776 .98368 L
F
.5752 .97887 m
.57776 .97887 L
.57776 .98368 L
.5752 .98368 L
F
.54186 .97887 m
.54443 .97887 L
.54443 .98368 L
.54186 .98368 L
F
.5393 .97887 m
.54186 .97887 L
.54186 .98368 L
.5393 .98368 L
F
.57263 .97887 m
.5752 .97887 L
.5752 .98368 L
.57263 .98368 L
F
.54955 .9933 m
.55212 .9933 L
.55212 .99811 L
.54955 .99811 L
F
.56494 .9933 m
.5675 .9933 L
.5675 .99811 L
.56494 .99811 L
F
.55725 .9933 m
.55981 .9933 L
.55981 .99811 L
.55725 .99811 L
F
.54699 .9933 m
.54955 .9933 L
.54955 .99811 L
.54699 .99811 L
F
.56238 .9933 m
.56494 .9933 L
.56494 .99811 L
.56238 .99811 L
F
.56494 .99811 m
.5675 .99811 L
.5675 1.00292 L
.56494 1.00292 L
F
.5675 .9933 m
.57007 .9933 L
.57007 .99811 L
.5675 .99811 L
F
.55981 .9933 m
.56238 .9933 L
.56238 .99811 L
.55981 .99811 L
F
.59058 .9933 m
.59314 .9933 L
.59314 .99811 L
.59058 .99811 L
F
.58289 .9933 m
.58545 .9933 L
.58545 .99811 L
.58289 .99811 L
F
.57263 .9933 m
.5752 .9933 L
.5752 .99811 L
.57263 .99811 L
F
.58802 .9933 m
.59058 .9933 L
.59058 .99811 L
.58802 .99811 L
F
.55468 .9933 m
.55725 .9933 L
.55725 .99811 L
.55468 .99811 L
F
.55212 .9933 m
.55468 .9933 L
.55468 .99811 L
.55212 .99811 L
F
.58545 .9933 m
.58802 .9933 L
.58802 .99811 L
.58545 .99811 L
F
.5393 .99811 m
.54186 .99811 L
.54186 1.00292 L
.5393 1.00292 L
F
.54443 .99811 m
.54699 .99811 L
.54699 1.00292 L
.54443 1.00292 L
F
.57263 .99811 m
.5752 .99811 L
.5752 1.00292 L
.57263 1.00292 L
F
.54186 .99811 m
.54443 .99811 L
.54443 1.00292 L
.54186 1.00292 L
F
.59571 .9933 m
.59827 .9933 L
.59827 .99811 L
.59571 .99811 L
F
.55212 .99811 m
.55468 .99811 L
.55468 1.00292 L
.55212 1.00292 L
F
.59314 .9933 m
.59571 .9933 L
.59571 .99811 L
.59314 .99811 L
F
.55725 .99811 m
.55981 .99811 L
.55981 1.00292 L
.55725 1.00292 L
F
.55981 .99811 m
.56238 .99811 L
.56238 1.00292 L
.55981 1.00292 L
F
.56238 .99811 m
.56494 .99811 L
.56494 1.00292 L
.56238 1.00292 L
F
.5675 .99811 m
.57007 .99811 L
.57007 1.00292 L
.5675 1.00292 L
F
.57776 .99811 m
.58032 .99811 L
.58032 1.00292 L
.57776 1.00292 L
F
.57007 .99811 m
.57263 .99811 L
.57263 1.00292 L
.57007 1.00292 L
F
.5752 .99811 m
.57776 .99811 L
.57776 1.00292 L
.5752 1.00292 L
F
.54443 1.00773 m
.54699 1.00773 L
.54699 1.01254 L
.54443 1.01254 L
F
.59314 1.00292 m
.59571 1.00292 L
.59571 1.00773 L
.59314 1.00773 L
F
.54186 1.00773 m
.54443 1.00773 L
.54443 1.01254 L
.54186 1.01254 L
F
.5393 1.00773 m
.54186 1.00773 L
.54186 1.01254 L
.5393 1.01254 L
F
.55468 1.00773 m
.55725 1.00773 L
.55725 1.01254 L
.55468 1.01254 L
F
.54699 1.00773 m
.54955 1.00773 L
.54955 1.01254 L
.54699 1.01254 L
F
.59571 1.00292 m
.59827 1.00292 L
.59827 1.00773 L
.59571 1.00773 L
F
.55212 1.00773 m
.55468 1.00773 L
.55468 1.01254 L
.55212 1.01254 L
F
.58289 .99811 m
.58545 .99811 L
.58545 1.00292 L
.58289 1.00292 L
F
.5393 1.00292 m
.54186 1.00292 L
.54186 1.00773 L
.5393 1.00773 L
F
.59058 .99811 m
.59314 .99811 L
.59314 1.00292 L
.59058 1.00292 L
F
.58032 .99811 m
.58289 .99811 L
.58289 1.00292 L
.58032 1.00292 L
F
.59314 .99811 m
.59571 .99811 L
.59571 1.00292 L
.59314 1.00292 L
F
.59571 .99811 m
.59827 .99811 L
.59827 1.00292 L
.59571 1.00292 L
F
.54186 1.00292 m
.54443 1.00292 L
.54443 1.00773 L
.54186 1.00773 L
F
.58545 1.00292 m
.58802 1.00292 L
.58802 1.00773 L
.58545 1.00773 L
F
.57776 .9933 m
.58032 .9933 L
.58032 .99811 L
.57776 .99811 L
F
.59058 1.00292 m
.59314 1.00292 L
.59314 1.00773 L
.59058 1.00773 L
F
.58289 1.00292 m
.58545 1.00292 L
.58545 1.00773 L
.58289 1.00773 L
F
.58032 .9933 m
.58289 .9933 L
.58289 .99811 L
.58032 .99811 L
F
.57007 .9933 m
.57263 .9933 L
.57263 .99811 L
.57007 .99811 L
F
.5752 .9933 m
.57776 .9933 L
.57776 .99811 L
.5752 .99811 L
F
.5752 1.00292 m
.57776 1.00292 L
.57776 1.00773 L
.5752 1.00773 L
F
.58032 1.00292 m
.58289 1.00292 L
.58289 1.00773 L
.58032 1.00773 L
F
.54955 1.00773 m
.55212 1.00773 L
.55212 1.01254 L
.54955 1.01254 L
F
.57776 1.00292 m
.58032 1.00292 L
.58032 1.00773 L
.57776 1.00773 L
F
.57263 1.00292 m
.5752 1.00292 L
.5752 1.00773 L
.57263 1.00773 L
F
.58802 1.00292 m
.59058 1.00292 L
.59058 1.00773 L
.58802 1.00773 L
F
.57007 1.00292 m
.57263 1.00292 L
.57263 1.00773 L
.57007 1.00773 L
F
.54699 .90191 m
.54955 .90191 L
.54955 .90672 L
.54699 .90672 L
F
.55468 .90191 m
.55725 .90191 L
.55725 .90672 L
.55468 .90672 L
F
.55981 .90191 m
.56238 .90191 L
.56238 .90672 L
.55981 .90672 L
F
.54955 .90191 m
.55212 .90191 L
.55212 .90672 L
.54955 .90672 L
F
.58032 .91153 m
.58289 .91153 L
.58289 .91634 L
.58032 .91634 L
F
.58545 .91153 m
.58802 .91153 L
.58802 .91634 L
.58545 .91634 L
F
.55212 .90191 m
.55468 .90191 L
.55468 .90672 L
.55212 .90672 L
F
.58289 .91153 m
.58545 .91153 L
.58545 .91634 L
.58289 .91634 L
F
.54699 .91153 m
.54955 .91153 L
.54955 .91634 L
.54699 .91634 L
F
.59571 .90672 m
.59827 .90672 L
.59827 .91153 L
.59571 .91153 L
F
.55212 .91153 m
.55468 .91153 L
.55468 .91634 L
.55212 .91634 L
F
.54443 .91153 m
.54699 .91153 L
.54699 .91634 L
.54443 .91634 L
F
.55725 .90191 m
.55981 .90191 L
.55981 .90672 L
.55725 .90672 L
F
.56238 .90191 m
.56494 .90191 L
.56494 .90672 L
.56238 .90672 L
F
.59314 .90672 m
.59571 .90672 L
.59571 .91153 L
.59314 .91153 L
F
.55725 .91153 m
.55981 .91153 L
.55981 .91634 L
.55725 .91634 L
F
.57007 .91153 m
.57263 .91153 L
.57263 .91634 L
.57007 .91634 L
F
.56238 .91153 m
.56494 .91153 L
.56494 .91634 L
.56238 .91634 L
F
.55468 .91153 m
.55725 .91153 L
.55725 .91634 L
.55468 .91634 L
F
.5675 .91153 m
.57007 .91153 L
.57007 .91634 L
.5675 .91634 L
F
.5675 .90672 m
.57007 .90672 L
.57007 .91153 L
.5675 .91153 L
F
.57263 .91153 m
.5752 .91153 L
.5752 .91634 L
.57263 .91634 L
F
.56494 .91153 m
.5675 .91153 L
.5675 .91634 L
.56494 .91634 L
F
.59314 .91153 m
.59571 .91153 L
.59571 .91634 L
.59314 .91634 L
F
.59571 .8971 m
.59827 .8971 L
.59827 .90191 L
.59571 .90191 L
F
.5752 .91153 m
.57776 .91153 L
.57776 .91634 L
.5752 .91634 L
F
.57776 .91153 m
.58032 .91153 L
.58032 .91634 L
.57776 .91634 L
F
.59058 .91153 m
.59314 .91153 L
.59314 .91634 L
.59058 .91634 L
F
.55981 .91153 m
.56238 .91153 L
.56238 .91634 L
.55981 .91634 L
F
.58802 .91153 m
.59058 .91153 L
.59058 .91634 L
.58802 .91634 L
F
.59314 .85862 m
.59571 .85862 L
.59571 .86343 L
.59314 .86343 L
F
.58545 .85862 m
.58802 .85862 L
.58802 .86343 L
.58545 .86343 L
F
.5393 .85862 m
.54186 .85862 L
.54186 .86343 L
.5393 .86343 L
F
.59058 .85862 m
.59314 .85862 L
.59314 .86343 L
.59058 .86343 L
F
.54443 .86343 m
.54699 .86343 L
.54699 .86824 L
.54443 .86824 L
F
.59571 .85862 m
.59827 .85862 L
.59827 .86343 L
.59571 .86343 L
F
.58802 .85862 m
.59058 .85862 L
.59058 .86343 L
.58802 .86343 L
F
.54186 .86343 m
.54443 .86343 L
.54443 .86824 L
.54186 .86824 L
F
.58545 .85381 m
.58802 .85381 L
.58802 .85862 L
.58545 .85862 L
F
.58802 .85381 m
.59058 .85381 L
.59058 .85862 L
.58802 .85862 L
F
.59058 .85381 m
.59314 .85381 L
.59314 .85862 L
.59058 .85862 L
F
.59314 .85381 m
.59571 .85381 L
.59571 .85862 L
.59314 .85862 L
F
.54443 .85862 m
.54699 .85862 L
.54699 .86343 L
.54443 .86343 L
F
.59571 .85381 m
.59827 .85381 L
.59827 .85862 L
.59571 .85862 L
F
.54186 .85862 m
.54443 .85862 L
.54443 .86343 L
.54186 .86343 L
F
.5393 .90191 m
.54186 .90191 L
.54186 .90672 L
.5393 .90672 L
F
.54186 .91153 m
.54443 .91153 L
.54443 .91634 L
.54186 .91634 L
F
.57776 .85862 m
.58032 .85862 L
.58032 .86343 L
.57776 .86343 L
F
.5393 .91153 m
.54186 .91153 L
.54186 .91634 L
.5393 .91634 L
F
.54186 .90191 m
.54443 .90191 L
.54443 .90672 L
.54186 .90672 L
F
.54955 .91153 m
.55212 .91153 L
.55212 .91634 L
.54955 .91634 L
F
.54443 .90191 m
.54699 .90191 L
.54699 .90672 L
.54443 .90672 L
F
.5752 .85862 m
.57776 .85862 L
.57776 .86343 L
.5752 .86343 L
F
.5675 .85862 m
.57007 .85862 L
.57007 .86343 L
.5675 .86343 L
F
.5393 .86343 m
.54186 .86343 L
.54186 .86824 L
.5393 .86824 L
F
.57263 .85862 m
.5752 .85862 L
.5752 .86343 L
.57263 .86343 L
F
.58032 .85862 m
.58289 .85862 L
.58289 .86343 L
.58032 .86343 L
F
.58289 .85862 m
.58545 .85862 L
.58545 .86343 L
.58289 .86343 L
F
.57007 .85862 m
.57263 .85862 L
.57263 .86343 L
.57007 .86343 L
F
.56494 .8971 m
.5675 .8971 L
.5675 .90191 L
.56494 .90191 L
F
.58545 .8971 m
.58802 .8971 L
.58802 .90191 L
.58545 .90191 L
F
.56238 .8971 m
.56494 .8971 L
.56494 .90191 L
.56238 .90191 L
F
.55981 .8971 m
.56238 .8971 L
.56238 .90191 L
.55981 .90191 L
F
.57776 .8971 m
.58032 .8971 L
.58032 .90191 L
.57776 .90191 L
F
.5752 .8971 m
.57776 .8971 L
.57776 .90191 L
.5752 .90191 L
F
.58289 .8971 m
.58545 .8971 L
.58545 .90191 L
.58289 .90191 L
F
.58802 .8971 m
.59058 .8971 L
.59058 .90191 L
.58802 .90191 L
F
.59571 .90191 m
.59827 .90191 L
.59827 .90672 L
.59571 .90672 L
F
.59314 .90191 m
.59571 .90191 L
.59571 .90672 L
.59314 .90672 L
F
.59058 .90191 m
.59314 .90191 L
.59314 .90672 L
.59058 .90672 L
F
.58802 .90191 m
.59058 .90191 L
.59058 .90672 L
.58802 .90672 L
F
.5675 .8971 m
.57007 .8971 L
.57007 .90191 L
.5675 .90191 L
F
.57263 .8971 m
.5752 .8971 L
.5752 .90191 L
.57263 .90191 L
F
.57007 .8971 m
.57263 .8971 L
.57263 .90191 L
.57007 .90191 L
F
.55725 .8971 m
.55981 .8971 L
.55981 .90191 L
.55725 .90191 L
F
.54699 .8971 m
.54955 .8971 L
.54955 .90191 L
.54699 .90191 L
F
.55468 .8971 m
.55725 .8971 L
.55725 .90191 L
.55468 .90191 L
F
.55212 .8971 m
.55468 .8971 L
.55468 .90191 L
.55212 .90191 L
F
.59314 .8971 m
.59571 .8971 L
.59571 .90191 L
.59314 .90191 L
F
.59058 .8971 m
.59314 .8971 L
.59314 .90191 L
.59058 .90191 L
F
.54955 .8971 m
.55212 .8971 L
.55212 .90191 L
.54955 .90191 L
F
.54186 .8971 m
.54443 .8971 L
.54443 .90191 L
.54186 .90191 L
F
.54443 .8971 m
.54699 .8971 L
.54699 .90191 L
.54443 .90191 L
F
.58032 .8971 m
.58289 .8971 L
.58289 .90191 L
.58032 .90191 L
F
.59571 .91153 m
.59827 .91153 L
.59827 .91634 L
.59571 .91634 L
F
.54443 .91634 m
.54699 .91634 L
.54699 .92115 L
.54443 .92115 L
F
.5393 .91634 m
.54186 .91634 L
.54186 .92115 L
.5393 .92115 L
F
.54186 .91634 m
.54443 .91634 L
.54443 .92115 L
.54186 .92115 L
F
.54699 .90672 m
.54955 .90672 L
.54955 .91153 L
.54699 .91153 L
F
.5393 .90672 m
.54186 .90672 L
.54186 .91153 L
.5393 .91153 L
F
.5752 .90672 m
.57776 .90672 L
.57776 .91153 L
.5752 .91153 L
F
.54443 .90672 m
.54699 .90672 L
.54699 .91153 L
.54443 .91153 L
F
.55725 .90672 m
.55981 .90672 L
.55981 .91153 L
.55725 .91153 L
F
.54955 .90672 m
.55212 .90672 L
.55212 .91153 L
.54955 .91153 L
F
.54186 .90672 m
.54443 .90672 L
.54443 .91153 L
.54186 .91153 L
F
.55468 .90672 m
.55725 .90672 L
.55725 .91153 L
.55468 .91153 L
F
.55981 .90672 m
.56238 .90672 L
.56238 .91153 L
.55981 .91153 L
F
.56238 .90672 m
.56494 .90672 L
.56494 .91153 L
.56238 .91153 L
F
.56494 .90672 m
.5675 .90672 L
.5675 .91153 L
.56494 .91153 L
F
.57007 .90672 m
.57263 .90672 L
.57263 .91153 L
.57007 .91153 L
F
.58032 .90672 m
.58289 .90672 L
.58289 .91153 L
.58032 .91153 L
F
.57263 .90672 m
.5752 .90672 L
.5752 .91153 L
.57263 .91153 L
F
.57776 .90672 m
.58032 .90672 L
.58032 .91153 L
.57776 .91153 L
F
.57007 .90191 m
.57263 .90191 L
.57263 .90672 L
.57007 .90672 L
F
.58802 .90672 m
.59058 .90672 L
.59058 .91153 L
.58802 .91153 L
F
.56494 .90191 m
.5675 .90191 L
.5675 .90672 L
.56494 .90672 L
F
.57263 .90191 m
.5752 .90191 L
.5752 .90672 L
.57263 .90672 L
F
.58289 .90672 m
.58545 .90672 L
.58545 .91153 L
.58289 .91153 L
F
.58545 .90672 m
.58802 .90672 L
.58802 .91153 L
.58545 .91153 L
F
.59058 .90672 m
.59314 .90672 L
.59314 .91153 L
.59058 .91153 L
F
.58032 .90191 m
.58289 .90191 L
.58289 .90672 L
.58032 .90672 L
F
.5752 .90191 m
.57776 .90191 L
.57776 .90672 L
.5752 .90672 L
F
.55212 .90672 m
.55468 .90672 L
.55468 .91153 L
.55212 .91153 L
F
.57776 .90191 m
.58032 .90191 L
.58032 .90672 L
.57776 .90672 L
F
.58289 .90191 m
.58545 .90191 L
.58545 .90672 L
.58289 .90672 L
F
.5675 .90191 m
.57007 .90191 L
.57007 .90672 L
.5675 .90672 L
F
.58545 .90191 m
.58802 .90191 L
.58802 .90672 L
.58545 .90672 L
F
.55212 .85381 m
.55468 .85381 L
.55468 .85862 L
.55212 .85862 L
F
.54443 .85381 m
.54699 .85381 L
.54699 .85862 L
.54443 .85862 L
F
.57776 .85381 m
.58032 .85381 L
.58032 .85862 L
.57776 .85862 L
F
.54955 .85381 m
.55212 .85381 L
.55212 .85862 L
.54955 .85862 L
F
.56238 .85381 m
.56494 .85381 L
.56494 .85862 L
.56238 .85862 L
F
.55468 .85381 m
.55725 .85381 L
.55725 .85862 L
.55468 .85862 L
F
.54699 .85381 m
.54955 .85381 L
.54955 .85862 L
.54699 .85862 L
F
.55981 .85381 m
.56238 .85381 L
.56238 .85862 L
.55981 .85862 L
F
.56494 .85381 m
.5675 .85381 L
.5675 .85862 L
.56494 .85862 L
F
.5675 .85381 m
.57007 .85381 L
.57007 .85862 L
.5675 .85862 L
F
.57007 .85381 m
.57263 .85381 L
.57263 .85862 L
.57007 .85862 L
F
.57263 .85381 m
.5752 .85381 L
.5752 .85862 L
.57263 .85862 L
F
.58289 .85381 m
.58545 .85381 L
.58545 .85862 L
.58289 .85862 L
F
.5752 .85381 m
.57776 .85381 L
.57776 .85862 L
.5752 .85862 L
F
.58032 .85381 m
.58289 .85381 L
.58289 .85862 L
.58032 .85862 L
F
.54443 .84418 m
.54699 .84418 L
.54699 .849 L
.54443 .849 L
F
.54699 .84418 m
.54955 .84418 L
.54955 .849 L
.54699 .849 L
F
.57007 .84418 m
.57263 .84418 L
.57263 .849 L
.57007 .849 L
F
.57263 .84418 m
.5752 .84418 L
.5752 .849 L
.57263 .849 L
F
.55468 .84418 m
.55725 .84418 L
.55725 .849 L
.55468 .849 L
F
.55725 .84418 m
.55981 .84418 L
.55981 .849 L
.55725 .849 L
F
.5393 .84418 m
.54186 .84418 L
.54186 .849 L
.5393 .849 L
F
.54186 .84418 m
.54443 .84418 L
.54443 .849 L
.54186 .849 L
F
.5675 .84418 m
.57007 .84418 L
.57007 .849 L
.5675 .849 L
F
.55981 .84418 m
.56238 .84418 L
.56238 .849 L
.55981 .849 L
F
.55725 .85381 m
.55981 .85381 L
.55981 .85862 L
.55725 .85862 L
F
.56494 .84418 m
.5675 .84418 L
.5675 .849 L
.56494 .849 L
F
.5752 .84418 m
.57776 .84418 L
.57776 .849 L
.5752 .849 L
F
.57776 .84418 m
.58032 .84418 L
.58032 .849 L
.57776 .849 L
F
.56238 .84418 m
.56494 .84418 L
.56494 .849 L
.56238 .849 L
F
.54186 .88748 m
.54443 .88748 L
.54443 .89229 L
.54186 .89229 L
F
.55212 .88748 m
.55468 .88748 L
.55468 .89229 L
.55212 .89229 L
F
.54699 .88748 m
.54955 .88748 L
.54955 .89229 L
.54699 .89229 L
F
.5393 .88748 m
.54186 .88748 L
.54186 .89229 L
.5393 .89229 L
F
.58802 .849 m
.59058 .849 L
.59058 .85381 L
.58802 .85381 L
F
.59058 .849 m
.59314 .849 L
.59314 .85381 L
.59058 .85381 L
F
.55468 .88748 m
.55725 .88748 L
.55725 .89229 L
.55468 .89229 L
F
.54955 .88748 m
.55212 .88748 L
.55212 .89229 L
.54955 .89229 L
F
.57263 .88748 m
.5752 .88748 L
.5752 .89229 L
.57263 .89229 L
F
.5752 .88748 m
.57776 .88748 L
.57776 .89229 L
.5752 .89229 L
F
.55725 .88748 m
.55981 .88748 L
.55981 .89229 L
.55725 .89229 L
F
.55981 .88748 m
.56238 .88748 L
.56238 .89229 L
.55981 .89229 L
F
.57007 .88748 m
.57263 .88748 L
.57263 .89229 L
.57007 .89229 L
F
.54443 .88748 m
.54699 .88748 L
.54699 .89229 L
.54443 .89229 L
F
.5675 .88748 m
.57007 .88748 L
.57007 .89229 L
.5675 .89229 L
F
.56494 .849 m
.5675 .849 L
.5675 .85381 L
.56494 .85381 L
F
.57776 .849 m
.58032 .849 L
.58032 .85381 L
.57776 .85381 L
F
.57007 .849 m
.57263 .849 L
.57263 .85381 L
.57007 .85381 L
F
.56238 .849 m
.56494 .849 L
.56494 .85381 L
.56238 .85381 L
F
.57263 .849 m
.5752 .849 L
.5752 .85381 L
.57263 .85381 L
F
.5752 .849 m
.57776 .849 L
.57776 .85381 L
.5752 .85381 L
F
.58032 .849 m
.58289 .849 L
.58289 .85381 L
.58032 .85381 L
F
.5393 .85381 m
.54186 .85381 L
.54186 .85862 L
.5393 .85862 L
F
.54186 .85381 m
.54443 .85381 L
.54443 .85862 L
.54186 .85862 L
F
.58289 .849 m
.58545 .849 L
.58545 .85381 L
.58289 .85381 L
F
.58545 .849 m
.58802 .849 L
.58802 .85381 L
.58545 .85381 L
F
.59571 .849 m
.59827 .849 L
.59827 .85381 L
.59571 .85381 L
F
.5675 .849 m
.57007 .849 L
.57007 .85381 L
.5675 .85381 L
F
.59314 .849 m
.59571 .849 L
.59571 .85381 L
.59314 .85381 L
F
.56494 .86824 m
.5675 .86824 L
.5675 .87305 L
.56494 .87305 L
F
.5675 .86824 m
.57007 .86824 L
.57007 .87305 L
.5675 .87305 L
F
.54955 .86824 m
.55212 .86824 L
.55212 .87305 L
.54955 .87305 L
F
.55212 .86824 m
.55468 .86824 L
.55468 .87305 L
.55212 .87305 L
F
.59314 .86343 m
.59571 .86343 L
.59571 .86824 L
.59314 .86824 L
F
.59571 .86343 m
.59827 .86343 L
.59827 .86824 L
.59571 .86824 L
F
.55981 .86824 m
.56238 .86824 L
.56238 .87305 L
.55981 .87305 L
F
.56238 .86824 m
.56494 .86824 L
.56494 .87305 L
.56238 .87305 L
F
.56494 .86343 m
.5675 .86343 L
.5675 .86824 L
.56494 .86824 L
F
.55725 .86343 m
.55981 .86343 L
.55981 .86824 L
.55725 .86824 L
F
.54955 .86343 m
.55212 .86343 L
.55212 .86824 L
.54955 .86824 L
F
.56238 .86343 m
.56494 .86343 L
.56494 .86824 L
.56238 .86824 L
F
.55468 .86824 m
.55725 .86824 L
.55725 .87305 L
.55468 .87305 L
F
.55725 .86824 m
.55981 .86824 L
.55981 .87305 L
.55725 .87305 L
F
.55981 .86343 m
.56238 .86343 L
.56238 .86824 L
.55981 .86824 L
F
.54955 .85862 m
.55212 .85862 L
.55212 .86343 L
.54955 .86343 L
F
.56238 .85862 m
.56494 .85862 L
.56494 .86343 L
.56238 .86343 L
F
.55468 .85862 m
.55725 .85862 L
.55725 .86343 L
.55468 .86343 L
F
.54699 .85862 m
.54955 .85862 L
.54955 .86343 L
.54699 .86343 L
F
.55725 .85862 m
.55981 .85862 L
.55981 .86343 L
.55725 .86343 L
F
.55981 .85862 m
.56238 .85862 L
.56238 .86343 L
.55981 .86343 L
F
.56494 .85862 m
.5675 .85862 L
.5675 .86343 L
.56494 .86343 L
F
.54443 .86824 m
.54699 .86824 L
.54699 .87305 L
.54443 .87305 L
F
.54699 .86824 m
.54955 .86824 L
.54955 .87305 L
.54699 .87305 L
F
.58802 .86343 m
.59058 .86343 L
.59058 .86824 L
.58802 .86824 L
F
.59058 .86343 m
.59314 .86343 L
.59314 .86824 L
.59058 .86824 L
F
.54186 .86824 m
.54443 .86824 L
.54443 .87305 L
.54186 .87305 L
F
.55212 .85862 m
.55468 .85862 L
.55468 .86343 L
.55212 .86343 L
F
.5393 .86824 m
.54186 .86824 L
.54186 .87305 L
.5393 .87305 L
F
.55468 .849 m
.55725 .849 L
.55725 .85381 L
.55468 .85381 L
F
.58545 .84418 m
.58802 .84418 L
.58802 .849 L
.58545 .849 L
F
.55981 .849 m
.56238 .849 L
.56238 .85381 L
.55981 .85381 L
F
.55212 .849 m
.55468 .849 L
.55468 .85381 L
.55212 .85381 L
F
.58289 .84418 m
.58545 .84418 L
.58545 .849 L
.58289 .849 L
F
.59571 .84418 m
.59827 .84418 L
.59827 .849 L
.59571 .849 L
F
.58802 .84418 m
.59058 .84418 L
.59058 .849 L
.58802 .849 L
F
.58032 .84418 m
.58289 .84418 L
.58289 .849 L
.58032 .849 L
F
.54699 .849 m
.54955 .849 L
.54955 .85381 L
.54699 .85381 L
F
.54955 .849 m
.55212 .849 L
.55212 .85381 L
.54955 .85381 L
F
.54955 .84418 m
.55212 .84418 L
.55212 .849 L
.54955 .849 L
F
.55212 .84418 m
.55468 .84418 L
.55468 .849 L
.55212 .849 L
F
.54443 .849 m
.54699 .849 L
.54699 .85381 L
.54443 .85381 L
F
.55725 .849 m
.55981 .849 L
.55981 .85381 L
.55725 .85381 L
F
.54186 .849 m
.54443 .849 L
.54443 .85381 L
.54186 .85381 L
F
.57776 .86343 m
.58032 .86343 L
.58032 .86824 L
.57776 .86824 L
F
.58032 .86343 m
.58289 .86343 L
.58289 .86824 L
.58032 .86824 L
F
.58289 .86343 m
.58545 .86343 L
.58545 .86824 L
.58289 .86824 L
F
.58545 .86343 m
.58802 .86343 L
.58802 .86824 L
.58545 .86824 L
F
.55468 .86343 m
.55725 .86343 L
.55725 .86824 L
.55468 .86824 L
F
.54699 .86343 m
.54955 .86343 L
.54955 .86824 L
.54699 .86824 L
F
.55212 .86343 m
.55468 .86343 L
.55468 .86824 L
.55212 .86824 L
F
.59314 .84418 m
.59571 .84418 L
.59571 .849 L
.59314 .849 L
F
.57263 .86343 m
.5752 .86343 L
.5752 .86824 L
.57263 .86824 L
F
.5393 .849 m
.54186 .849 L
.54186 .85381 L
.5393 .85381 L
F
.59058 .84418 m
.59314 .84418 L
.59314 .849 L
.59058 .849 L
F
.5675 .86343 m
.57007 .86343 L
.57007 .86824 L
.5675 .86824 L
F
.57007 .86343 m
.57263 .86343 L
.57263 .86824 L
.57007 .86824 L
F
.5752 .86343 m
.57776 .86343 L
.57776 .86824 L
.5752 .86824 L
F
.55981 .92596 m
.56238 .92596 L
.56238 .93077 L
.55981 .93077 L
F
.56238 .92596 m
.56494 .92596 L
.56494 .93077 L
.56238 .93077 L
F
.56494 .92596 m
.5675 .92596 L
.5675 .93077 L
.56494 .93077 L
F
.57007 .92596 m
.57263 .92596 L
.57263 .93077 L
.57007 .93077 L
F
.59314 .92115 m
.59571 .92115 L
.59571 .92596 L
.59314 .92596 L
F
.58289 .92115 m
.58545 .92115 L
.58545 .92596 L
.58289 .92596 L
F
.59058 .92115 m
.59314 .92115 L
.59314 .92596 L
.59058 .92596 L
F
.58802 .92115 m
.59058 .92115 L
.59058 .92596 L
.58802 .92596 L
F
.55212 .92596 m
.55468 .92596 L
.55468 .93077 L
.55212 .93077 L
F
.55725 .92596 m
.55981 .92596 L
.55981 .93077 L
.55725 .93077 L
F
.58545 .96444 m
.58802 .96444 L
.58802 .96925 L
.58545 .96925 L
F
.55468 .92596 m
.55725 .92596 L
.55725 .93077 L
.55468 .93077 L
F
.54955 .92596 m
.55212 .92596 L
.55212 .93077 L
.54955 .93077 L
F
.5675 .92596 m
.57007 .92596 L
.57007 .93077 L
.5675 .93077 L
F
.54699 .92596 m
.54955 .92596 L
.54955 .93077 L
.54699 .93077 L
F
.5393 .93077 m
.54186 .93077 L
.54186 .93558 L
.5393 .93558 L
F
.55468 .93077 m
.55725 .93077 L
.55725 .93558 L
.55468 .93558 L
F
.54699 .93077 m
.54955 .93077 L
.54955 .93558 L
.54699 .93558 L
F
.59571 .92596 m
.59827 .92596 L
.59827 .93077 L
.59571 .93077 L
F
.55212 .93077 m
.55468 .93077 L
.55468 .93558 L
.55212 .93558 L
F
.58032 .92596 m
.58289 .92596 L
.58289 .93077 L
.58032 .93077 L
F
.55725 .93077 m
.55981 .93077 L
.55981 .93558 L
.55725 .93558 L
F
.54955 .93077 m
.55212 .93077 L
.55212 .93558 L
.54955 .93558 L
F
.54443 .92596 m
.54699 .92596 L
.54699 .93077 L
.54443 .93077 L
F
.59571 .92115 m
.59827 .92115 L
.59827 .92596 L
.59571 .92596 L
F
.58545 .92115 m
.58802 .92115 L
.58802 .92596 L
.58545 .92596 L
F
.54186 .92596 m
.54443 .92596 L
.54443 .93077 L
.54186 .93077 L
F
.54443 .93077 m
.54699 .93077 L
.54699 .93558 L
.54443 .93558 L
F
.54186 .93077 m
.54443 .93077 L
.54443 .93558 L
.54186 .93558 L
F
.5393 .92596 m
.54186 .92596 L
.54186 .93077 L
.5393 .93077 L
F
.54443 .96444 m
.54699 .96444 L
.54699 .96925 L
.54443 .96925 L
F
.59571 .95963 m
.59827 .95963 L
.59827 .96444 L
.59571 .96444 L
F
.58545 .95963 m
.58802 .95963 L
.58802 .96444 L
.58545 .96444 L
F
.54186 .96444 m
.54443 .96444 L
.54443 .96925 L
.54186 .96925 L
F
.5393 .96925 m
.54186 .96925 L
.54186 .97406 L
.5393 .97406 L
F
.54443 .96925 m
.54699 .96925 L
.54699 .97406 L
.54443 .97406 L
F
.5393 .96444 m
.54186 .96444 L
.54186 .96925 L
.5393 .96925 L
F
.54186 .96925 m
.54443 .96925 L
.54443 .97406 L
.54186 .97406 L
F
.56238 .96444 m
.56494 .96444 L
.56494 .96925 L
.56238 .96925 L
F
.59058 .95963 m
.59314 .95963 L
.59314 .96444 L
.59058 .96444 L
F
.5675 .96444 m
.57007 .96444 L
.57007 .96925 L
.5675 .96925 L
F
.55981 .96444 m
.56238 .96444 L
.56238 .96925 L
.55981 .96925 L
F
.59314 .95963 m
.59571 .95963 L
.59571 .96444 L
.59314 .96444 L
F
.58289 .95963 m
.58545 .95963 L
.58545 .96444 L
.58289 .96444 L
F
.58802 .95963 m
.59058 .95963 L
.59058 .96444 L
.58802 .96444 L
F
.57007 .96444 m
.57263 .96444 L
.57263 .96925 L
.57007 .96925 L
F
.57263 .96444 m
.5752 .96444 L
.5752 .96925 L
.57263 .96925 L
F
.5752 .96444 m
.57776 .96444 L
.57776 .96925 L
.5752 .96925 L
F
.58032 .96444 m
.58289 .96444 L
.58289 .96925 L
.58032 .96925 L
F
.59058 .96444 m
.59314 .96444 L
.59314 .96925 L
.59058 .96925 L
F
.58289 .96444 m
.58545 .96444 L
.58545 .96925 L
.58289 .96925 L
F
.58802 .96444 m
.59058 .96444 L
.59058 .96925 L
.58802 .96925 L
F
.55212 .96925 m
.55468 .96925 L
.55468 .97406 L
.55212 .97406 L
F
.55468 .96925 m
.55725 .96925 L
.55725 .97406 L
.55468 .97406 L
F
.59314 .96444 m
.59571 .96444 L
.59571 .96925 L
.59314 .96925 L
F
.59571 .96444 m
.59827 .96444 L
.59827 .96925 L
.59571 .96925 L
F
.54955 .96925 m
.55212 .96925 L
.55212 .97406 L
.54955 .97406 L
F
.57776 .96444 m
.58032 .96444 L
.58032 .96925 L
.57776 .96925 L
F
.54699 .96925 m
.54955 .96925 L
.54955 .97406 L
.54699 .97406 L
F
.54699 .92115 m
.54955 .92115 L
.54955 .92596 L
.54699 .92596 L
F
.59571 .91634 m
.59827 .91634 L
.59827 .92115 L
.59571 .92115 L
F
.54443 .92115 m
.54699 .92115 L
.54699 .92596 L
.54443 .92596 L
F
.54186 .92115 m
.54443 .92115 L
.54443 .92596 L
.54186 .92596 L
F
.55725 .92115 m
.55981 .92115 L
.55981 .92596 L
.55725 .92596 L
F
.54955 .92115 m
.55212 .92115 L
.55212 .92596 L
.54955 .92596 L
F
.5393 .92115 m
.54186 .92115 L
.54186 .92596 L
.5393 .92596 L
F
.55468 .92115 m
.55725 .92115 L
.55725 .92596 L
.55468 .92596 L
F
.56238 .92115 m
.56494 .92115 L
.56494 .92596 L
.56238 .92596 L
F
.57776 .92115 m
.58032 .92115 L
.58032 .92596 L
.57776 .92596 L
F
.57007 .92115 m
.57263 .92115 L
.57263 .92596 L
.57007 .92596 L
F
.55981 .92115 m
.56238 .92115 L
.56238 .92596 L
.55981 .92596 L
F
.57263 .92115 m
.5752 .92115 L
.5752 .92596 L
.57263 .92596 L
F
.5752 .92115 m
.57776 .92115 L
.57776 .92596 L
.5752 .92596 L
F
.58032 .92115 m
.58289 .92115 L
.58289 .92596 L
.58032 .92596 L
F
.57007 .94039 m
.57263 .94039 L
.57263 .9452 L
.57007 .9452 L
F
.57263 .94039 m
.5752 .94039 L
.5752 .9452 L
.57263 .9452 L
F
.5752 .94039 m
.57776 .94039 L
.57776 .9452 L
.5752 .9452 L
F
.58032 .94039 m
.58289 .94039 L
.58289 .9452 L
.58032 .9452 L
F
.5393 .94039 m
.54186 .94039 L
.54186 .9452 L
.5393 .9452 L
F
.54443 .94039 m
.54699 .94039 L
.54699 .9452 L
.54443 .9452 L
F
.54186 .94039 m
.54443 .94039 L
.54443 .9452 L
.54186 .9452 L
F
.56238 .94039 m
.56494 .94039 L
.56494 .9452 L
.56238 .9452 L
F
.5675 .94039 m
.57007 .94039 L
.57007 .9452 L
.5675 .9452 L
F
.55212 .92115 m
.55468 .92115 L
.55468 .92596 L
.55212 .92596 L
F
.56494 .94039 m
.5675 .94039 L
.5675 .9452 L
.56494 .9452 L
F
.55981 .94039 m
.56238 .94039 L
.56238 .9452 L
.55981 .9452 L
F
.57776 .94039 m
.58032 .94039 L
.58032 .9452 L
.57776 .9452 L
F
.55725 .94039 m
.55981 .94039 L
.55981 .9452 L
.55725 .9452 L
F
.5752 .91634 m
.57776 .91634 L
.57776 .92115 L
.5752 .92115 L
F
.58032 .91634 m
.58289 .91634 L
.58289 .92115 L
.58032 .92115 L
F
.58802 .92596 m
.59058 .92596 L
.59058 .93077 L
.58802 .93077 L
F
.57776 .91634 m
.58032 .91634 L
.58032 .92115 L
.57776 .92115 L
F
.59058 .91634 m
.59314 .91634 L
.59314 .92115 L
.59058 .92115 L
F
.58802 .91634 m
.59058 .91634 L
.59058 .92115 L
.58802 .92115 L
F
.57007 .91634 m
.57263 .91634 L
.57263 .92115 L
.57007 .92115 L
F
.57263 .91634 m
.5752 .91634 L
.5752 .92115 L
.57263 .92115 L
F
.57263 .92596 m
.5752 .92596 L
.5752 .93077 L
.57263 .93077 L
F
.5752 .92596 m
.57776 .92596 L
.57776 .93077 L
.5752 .93077 L
F
.57776 .92596 m
.58032 .92596 L
.58032 .93077 L
.57776 .93077 L
F
.58289 .92596 m
.58545 .92596 L
.58545 .93077 L
.58289 .93077 L
F
.59314 .92596 m
.59571 .92596 L
.59571 .93077 L
.59314 .93077 L
F
.58545 .92596 m
.58802 .92596 L
.58802 .93077 L
.58545 .93077 L
F
.59058 .92596 m
.59314 .92596 L
.59314 .93077 L
.59058 .93077 L
F
.5675 .91634 m
.57007 .91634 L
.57007 .92115 L
.5675 .92115 L
F
.55981 .91634 m
.56238 .91634 L
.56238 .92115 L
.55981 .92115 L
F
.54955 .91634 m
.55212 .91634 L
.55212 .92115 L
.54955 .92115 L
F
.56494 .91634 m
.5675 .91634 L
.5675 .92115 L
.56494 .92115 L
F
.5675 .92115 m
.57007 .92115 L
.57007 .92596 L
.5675 .92596 L
F
.56494 .92115 m
.5675 .92115 L
.5675 .92596 L
.56494 .92596 L
F
.56238 .91634 m
.56494 .91634 L
.56494 .92115 L
.56238 .92115 L
F
.58545 .91634 m
.58802 .91634 L
.58802 .92115 L
.58545 .92115 L
F
.55468 .91634 m
.55725 .91634 L
.55725 .92115 L
.55468 .92115 L
F
.59314 .91634 m
.59571 .91634 L
.59571 .92115 L
.59314 .92115 L
F
.58289 .91634 m
.58545 .91634 L
.58545 .92115 L
.58289 .92115 L
F
.55725 .91634 m
.55981 .91634 L
.55981 .92115 L
.55725 .92115 L
F
.54699 .91634 m
.54955 .91634 L
.54955 .92115 L
.54699 .92115 L
F
.55212 .91634 m
.55468 .91634 L
.55468 .92115 L
.55212 .92115 L
F
.5393 .95001 m
.54186 .95001 L
.54186 .95482 L
.5393 .95482 L
F
.54443 .95001 m
.54699 .95001 L
.54699 .95482 L
.54443 .95482 L
F
.55212 .95963 m
.55468 .95963 L
.55468 .96444 L
.55212 .96444 L
F
.54186 .95001 m
.54443 .95001 L
.54443 .95482 L
.54186 .95482 L
F
.55468 .95001 m
.55725 .95001 L
.55725 .95482 L
.55468 .95482 L
F
.55212 .95001 m
.55468 .95001 L
.55468 .95482 L
.55212 .95482 L
F
.59314 .9452 m
.59571 .9452 L
.59571 .95001 L
.59314 .95001 L
F
.59571 .9452 m
.59827 .9452 L
.59827 .95001 L
.59571 .95001 L
F
.59571 .95482 m
.59827 .95482 L
.59827 .95963 L
.59571 .95963 L
F
.5393 .95963 m
.54186 .95963 L
.54186 .96444 L
.5393 .96444 L
F
.54186 .95963 m
.54443 .95963 L
.54443 .96444 L
.54186 .96444 L
F
.54699 .95963 m
.54955 .95963 L
.54955 .96444 L
.54699 .96444 L
F
.55725 .95963 m
.55981 .95963 L
.55981 .96444 L
.55725 .96444 L
F
.54955 .95963 m
.55212 .95963 L
.55212 .96444 L
.54955 .96444 L
F
.55468 .95963 m
.55725 .95963 L
.55725 .96444 L
.55468 .96444 L
F
.59058 .9452 m
.59314 .9452 L
.59314 .95001 L
.59058 .95001 L
F
.58289 .9452 m
.58545 .9452 L
.58545 .95001 L
.58289 .95001 L
F
.57263 .9452 m
.5752 .9452 L
.5752 .95001 L
.57263 .95001 L
F
.58802 .9452 m
.59058 .9452 L
.59058 .95001 L
.58802 .95001 L
F
.58802 .95001 m
.59058 .95001 L
.59058 .95482 L
.58802 .95482 L
F
.59314 .95001 m
.59571 .95001 L
.59571 .95482 L
.59314 .95482 L
F
.58545 .9452 m
.58802 .9452 L
.58802 .95001 L
.58545 .95001 L
F
.59058 .95001 m
.59314 .95001 L
.59314 .95482 L
.59058 .95482 L
F
.54955 .95001 m
.55212 .95001 L
.55212 .95482 L
.54955 .95482 L
F
.57776 .9452 m
.58032 .9452 L
.58032 .95001 L
.57776 .95001 L
F
.55725 .95001 m
.55981 .95001 L
.55981 .95482 L
.55725 .95482 L
F
.54699 .95001 m
.54955 .95001 L
.54955 .95482 L
.54699 .95482 L
F
.58032 .9452 m
.58289 .9452 L
.58289 .95001 L
.58032 .95001 L
F
.57007 .9452 m
.57263 .9452 L
.57263 .95001 L
.57007 .95001 L
F
.5752 .9452 m
.57776 .9452 L
.57776 .95001 L
.5752 .95001 L
F
.58545 .95482 m
.58802 .95482 L
.58802 .95963 L
.58545 .95963 L
F
.55468 .95482 m
.55725 .95482 L
.55725 .95963 L
.55468 .95963 L
F
.59314 .95482 m
.59571 .95482 L
.59571 .95963 L
.59314 .95963 L
F
.58289 .95482 m
.58545 .95482 L
.58545 .95963 L
.58289 .95963 L
F
.54699 .95482 m
.54955 .95482 L
.54955 .95963 L
.54699 .95963 L
F
.54955 .95482 m
.55212 .95482 L
.55212 .95963 L
.54955 .95963 L
F
.55212 .95482 m
.55468 .95482 L
.55468 .95963 L
.55212 .95963 L
F
.55725 .95482 m
.55981 .95482 L
.55981 .95963 L
.55725 .95963 L
F
.58032 .95482 m
.58289 .95482 L
.58289 .95963 L
.58032 .95963 L
F
.57007 .95482 m
.57263 .95482 L
.57263 .95963 L
.57007 .95963 L
F
.57776 .95482 m
.58032 .95482 L
.58032 .95963 L
.57776 .95963 L
F
.5752 .95482 m
.57776 .95482 L
.57776 .95963 L
.5752 .95963 L
F
.59058 .95482 m
.59314 .95482 L
.59314 .95963 L
.59058 .95963 L
F
.58802 .95482 m
.59058 .95482 L
.59058 .95963 L
.58802 .95963 L
F
.57263 .95482 m
.5752 .95482 L
.5752 .95963 L
.57263 .95963 L
F
.57776 .95963 m
.58032 .95963 L
.58032 .96444 L
.57776 .96444 L
F
.58032 .95963 m
.58289 .95963 L
.58289 .96444 L
.58032 .96444 L
F
.55981 .95963 m
.56238 .95963 L
.56238 .96444 L
.55981 .96444 L
F
.56238 .95963 m
.56494 .95963 L
.56494 .96444 L
.56238 .96444 L
F
.5752 .95963 m
.57776 .95963 L
.57776 .96444 L
.5752 .96444 L
F
.54443 .95963 m
.54699 .95963 L
.54699 .96444 L
.54443 .96444 L
F
.57263 .95963 m
.5752 .95963 L
.5752 .96444 L
.57263 .96444 L
F
.55981 .95482 m
.56238 .95482 L
.56238 .95963 L
.55981 .95963 L
F
.56238 .95482 m
.56494 .95482 L
.56494 .95963 L
.56238 .95963 L
F
.56494 .95482 m
.5675 .95482 L
.5675 .95963 L
.56494 .95963 L
F
.5675 .95482 m
.57007 .95482 L
.57007 .95963 L
.5675 .95963 L
F
.56494 .95963 m
.5675 .95963 L
.5675 .96444 L
.56494 .96444 L
F
.57007 .95963 m
.57263 .95963 L
.57263 .96444 L
.57007 .96444 L
F
.5675 .95963 m
.57007 .95963 L
.57007 .96444 L
.5675 .96444 L
F
.57263 .97406 m
.5752 .97406 L
.5752 .97887 L
.57263 .97887 L
F
.58802 .97406 m
.59058 .97406 L
.59058 .97887 L
.58802 .97887 L
F
.58032 .97406 m
.58289 .97406 L
.58289 .97887 L
.58032 .97887 L
F
.57007 .97406 m
.57263 .97406 L
.57263 .97887 L
.57007 .97887 L
F
.58545 .97406 m
.58802 .97406 L
.58802 .97887 L
.58545 .97887 L
F
.55468 .97406 m
.55725 .97406 L
.55725 .97887 L
.55468 .97887 L
F
.59058 .97406 m
.59314 .97406 L
.59314 .97887 L
.59058 .97887 L
F
.58289 .97406 m
.58545 .97406 L
.58545 .97887 L
.58289 .97887 L
F
.57776 .96925 m
.58032 .96925 L
.58032 .97406 L
.57776 .97406 L
F
.57007 .96925 m
.57263 .96925 L
.57263 .97406 L
.57007 .97406 L
F
.55981 .96925 m
.56238 .96925 L
.56238 .97406 L
.55981 .97406 L
F
.5752 .96925 m
.57776 .96925 L
.57776 .97406 L
.5752 .97406 L
F
.57776 .97406 m
.58032 .97406 L
.58032 .97887 L
.57776 .97887 L
F
.5752 .97406 m
.57776 .97406 L
.57776 .97887 L
.5752 .97887 L
F
.57263 .96925 m
.5752 .96925 L
.5752 .97406 L
.57263 .97406 L
F
.55212 .96444 m
.55468 .96444 L
.55468 .96925 L
.55212 .96925 L
F
.55725 .96444 m
.55981 .96444 L
.55981 .96925 L
.55725 .96925 L
F
.56238 .97406 m
.56494 .97406 L
.56494 .97887 L
.56238 .97887 L
F
.55468 .96444 m
.55725 .96444 L
.55725 .96925 L
.55468 .96925 L
F
.54955 .96444 m
.55212 .96444 L
.55212 .96925 L
.54955 .96925 L
F
.56494 .96444 m
.5675 .96444 L
.5675 .96925 L
.56494 .96925 L
F
.54699 .96444 m
.54955 .96444 L
.54955 .96925 L
.54699 .96925 L
F
.54699 .97406 m
.54955 .97406 L
.54955 .97887 L
.54699 .97887 L
F
.54955 .97406 m
.55212 .97406 L
.55212 .97887 L
.54955 .97887 L
F
.55212 .97406 m
.55468 .97406 L
.55468 .97887 L
.55212 .97887 L
F
.55725 .97406 m
.55981 .97406 L
.55981 .97887 L
.55725 .97887 L
F
.5675 .97406 m
.57007 .97406 L
.57007 .97887 L
.5675 .97887 L
F
.55981 .97406 m
.56238 .97406 L
.56238 .97887 L
.55981 .97887 L
F
.56494 .97406 m
.5675 .97406 L
.5675 .97887 L
.56494 .97887 L
F
.55981 .95001 m
.56238 .95001 L
.56238 .95482 L
.55981 .95482 L
F
.56238 .95001 m
.56494 .95001 L
.56494 .95482 L
.56238 .95482 L
F
.56494 .95001 m
.5675 .95001 L
.5675 .95482 L
.56494 .95482 L
F
.57007 .95001 m
.57263 .95001 L
.57263 .95482 L
.57007 .95482 L
F
.57263 .95001 m
.5752 .95001 L
.5752 .95482 L
.57263 .95482 L
F
.5752 .95001 m
.57776 .95001 L
.57776 .95482 L
.5752 .95482 L
F
.57776 .95001 m
.58032 .95001 L
.58032 .95482 L
.57776 .95482 L
F
.58032 .95001 m
.58289 .95001 L
.58289 .95482 L
.58032 .95482 L
F
.54186 .95482 m
.54443 .95482 L
.54443 .95963 L
.54186 .95963 L
F
.54443 .95482 m
.54699 .95482 L
.54699 .95963 L
.54443 .95963 L
F
.58289 .95001 m
.58545 .95001 L
.58545 .95482 L
.58289 .95482 L
F
.58545 .95001 m
.58802 .95001 L
.58802 .95482 L
.58545 .95482 L
F
.5393 .95482 m
.54186 .95482 L
.54186 .95963 L
.5393 .95963 L
F
.5675 .95001 m
.57007 .95001 L
.57007 .95482 L
.5675 .95482 L
F
.59571 .95001 m
.59827 .95001 L
.59827 .95482 L
.59571 .95482 L
F
.59571 .96925 m
.59827 .96925 L
.59827 .97406 L
.59571 .97406 L
F
.56494 .96925 m
.5675 .96925 L
.5675 .97406 L
.56494 .97406 L
F
.54443 .97406 m
.54699 .97406 L
.54699 .97887 L
.54443 .97887 L
F
.59314 .96925 m
.59571 .96925 L
.59571 .97406 L
.59314 .97406 L
F
.5675 .96925 m
.57007 .96925 L
.57007 .97406 L
.5675 .97406 L
F
.55725 .96925 m
.55981 .96925 L
.55981 .97406 L
.55725 .97406 L
F
.56238 .96925 m
.56494 .96925 L
.56494 .97406 L
.56238 .97406 L
F
.59058 .96925 m
.59314 .96925 L
.59314 .97406 L
.59058 .97406 L
F
.58032 .96925 m
.58289 .96925 L
.58289 .97406 L
.58032 .97406 L
F
.58802 .96925 m
.59058 .96925 L
.59058 .97406 L
.58802 .97406 L
F
.58545 .96925 m
.58802 .96925 L
.58802 .97406 L
.58545 .97406 L
F
.54186 .97406 m
.54443 .97406 L
.54443 .97887 L
.54186 .97887 L
F
.5393 .97406 m
.54186 .97406 L
.54186 .97887 L
.5393 .97887 L
F
.58289 .96925 m
.58545 .96925 L
.58545 .97406 L
.58289 .97406 L
F
.59922 1.00773 m
.59922 1.01254 L
.59827 1.01254 L
.59827 1.00773 L
F
.59922 .98849 m
.59922 .9933 L
.59827 .9933 L
.59827 .98849 L
F
.59922 1.00292 m
.59922 1.00773 L
.59827 1.00773 L
.59827 1.00292 L
F
.59922 .99811 m
.59922 1.00292 L
.59827 1.00292 L
.59827 .99811 L
F
.59922 1.02697 m
.59922 1.03178 L
.59827 1.03178 L
.59827 1.02697 L
F
.59922 1.01254 m
.59922 1.01735 L
.59827 1.01735 L
.59827 1.01254 L
F
.59922 .9933 m
.59922 .99811 L
.59827 .99811 L
.59827 .9933 L
F
.59922 1.02216 m
.59922 1.02697 L
.59827 1.02697 L
.59827 1.02216 L
F
.59922 .85862 m
.59922 .86343 L
.59827 .86343 L
.59827 .85862 L
F
.59922 .88748 m
.59922 .89229 L
.59827 .89229 L
.59827 .88748 L
F
.59922 .87305 m
.59922 .87786 L
.59827 .87786 L
.59827 .87305 L
F
.59922 .85381 m
.59922 .85862 L
.59827 .85862 L
.59827 .85381 L
F
.59922 .87786 m
.59922 .88267 L
.59827 .88267 L
.59827 .87786 L
F
.59922 .88267 m
.59922 .88748 L
.59827 .88748 L
.59827 .88267 L
F
.59922 .89229 m
.59922 .8971 L
.59827 .8971 L
.59827 .89229 L
F
.59922 .97406 m
.59922 .97887 L
.59827 .97887 L
.59827 .97406 L
F
.54186 1.02697 m
.54443 1.02697 L
.54443 1.03178 L
.54186 1.03178 L
F
.59922 .98368 m
.59922 .98849 L
.59827 .98849 L
.59827 .98368 L
F
.59922 .96925 m
.59922 .97406 L
.59827 .97406 L
.59827 .96925 L
F
.59314 1.02216 m
.59571 1.02216 L
.59571 1.02697 L
.59314 1.02697 L
F
.59571 1.02216 m
.59827 1.02216 L
.59827 1.02697 L
.59571 1.02697 L
F
.5393 1.02697 m
.54186 1.02697 L
.54186 1.03178 L
.5393 1.03178 L
F
.54443 1.02697 m
.54699 1.02697 L
.54699 1.03178 L
.54443 1.03178 L
F
.59922 .95482 m
.59922 .95963 L
.59827 .95963 L
.59827 .95482 L
F
.59922 .96444 m
.59922 .96925 L
.59827 .96925 L
.59827 .96444 L
F
.59922 1.01735 m
.59922 1.02216 L
.59827 1.02216 L
.59827 1.01735 L
F
.59922 .95963 m
.59922 .96444 L
.59827 .96444 L
.59827 .95963 L
F
.59922 .95001 m
.59922 .95482 L
.59827 .95482 L
.59827 .95001 L
F
.59922 .97887 m
.59922 .98368 L
.59827 .98368 L
.59827 .97887 L
F
.59922 .9452 m
.59922 .95001 L
.59827 .95001 L
.59827 .9452 L
F
.55212 1.01254 m
.55468 1.01254 L
.55468 1.01735 L
.55212 1.01735 L
F
.55725 1.01254 m
.55981 1.01254 L
.55981 1.01735 L
.55725 1.01735 L
F
.58545 1.01254 m
.58802 1.01254 L
.58802 1.01735 L
.58545 1.01735 L
F
.55468 1.01254 m
.55725 1.01254 L
.55725 1.01735 L
.55468 1.01735 L
F
.56494 1.01254 m
.5675 1.01254 L
.5675 1.01735 L
.56494 1.01735 L
F
.5675 1.01254 m
.57007 1.01254 L
.57007 1.01735 L
.5675 1.01735 L
F
.54699 1.01254 m
.54955 1.01254 L
.54955 1.01735 L
.54699 1.01735 L
F
.54955 1.01254 m
.55212 1.01254 L
.55212 1.01735 L
.54955 1.01735 L
F
.57007 1.01254 m
.57263 1.01254 L
.57263 1.01735 L
.57007 1.01735 L
F
.57263 1.01254 m
.5752 1.01254 L
.5752 1.01735 L
.57263 1.01735 L
F
.5752 1.01254 m
.57776 1.01254 L
.57776 1.01735 L
.5752 1.01735 L
F
.58032 1.01254 m
.58289 1.01254 L
.58289 1.01735 L
.58032 1.01735 L
F
.59058 1.01254 m
.59314 1.01254 L
.59314 1.01735 L
.59058 1.01735 L
F
.58289 1.01254 m
.58545 1.01254 L
.58545 1.01735 L
.58289 1.01735 L
F
.58802 1.01254 m
.59058 1.01254 L
.59058 1.01735 L
.58802 1.01735 L
F
.59922 .94039 m
.59922 .9452 L
.59827 .9452 L
.59827 .94039 L
F
.59922 .92115 m
.59922 .92596 L
.59827 .92596 L
.59827 .92115 L
F
.59922 .93558 m
.59922 .94039 L
.59827 .94039 L
.59827 .93558 L
F
.59922 .93077 m
.59922 .93558 L
.59827 .93558 L
.59827 .93077 L
F
.59922 .86824 m
.59922 .87305 L
.59827 .87305 L
.59827 .86824 L
F
.59922 .86343 m
.59922 .86824 L
.59827 .86824 L
.59827 .86343 L
F
.59922 .92596 m
.59922 .93077 L
.59827 .93077 L
.59827 .92596 L
F
.59922 .91153 m
.59922 .91634 L
.59827 .91634 L
.59827 .91153 L
F
.59922 .90672 m
.59922 .91153 L
.59827 .91153 L
.59827 .90672 L
F
.55981 1.01254 m
.56238 1.01254 L
.56238 1.01735 L
.55981 1.01735 L
F
.56238 1.01254 m
.56494 1.01254 L
.56494 1.01735 L
.56238 1.01735 L
F
.59922 .8971 m
.59922 .90191 L
.59827 .90191 L
.59827 .8971 L
F
.59922 .90191 m
.59922 .90672 L
.59827 .90672 L
.59827 .90191 L
F
.59922 .91634 m
.59922 .92115 L
.59827 .92115 L
.59827 .91634 L
F
.55725 .98849 m
.55981 .98849 L
.55981 .9933 L
.55725 .9933 L
F
.54699 .98849 m
.54955 .98849 L
.54955 .9933 L
.54699 .9933 L
F
.55468 .98849 m
.55725 .98849 L
.55725 .9933 L
.55468 .9933 L
F
.55212 .98849 m
.55468 .98849 L
.55468 .9933 L
.55212 .9933 L
F
.5752 .98368 m
.57776 .98368 L
.57776 .98849 L
.5752 .98849 L
F
.58032 .98368 m
.58289 .98368 L
.58289 .98849 L
.58032 .98849 L
F
.54955 .98849 m
.55212 .98849 L
.55212 .9933 L
.54955 .9933 L
F
.57776 .98368 m
.58032 .98368 L
.58032 .98849 L
.57776 .98849 L
F
.54186 .98849 m
.54443 .98849 L
.54443 .9933 L
.54186 .9933 L
F
.5393 .98849 m
.54186 .98849 L
.54186 .9933 L
.5393 .9933 L
F
.57007 1.02697 m
.57263 1.02697 L
.57263 1.03178 L
.57007 1.03178 L
F
.57263 1.02697 m
.5752 1.02697 L
.5752 1.03178 L
.57263 1.03178 L
F
.59314 .98368 m
.59571 .98368 L
.59571 .98849 L
.59314 .98849 L
F
.59571 .98368 m
.59827 .98368 L
.59827 .98849 L
.59571 .98849 L
F
.54443 .98849 m
.54699 .98849 L
.54699 .9933 L
.54443 .9933 L
F
.58289 .98849 m
.58545 .98849 L
.58545 .9933 L
.58289 .9933 L
F
.58545 .98849 m
.58802 .98849 L
.58802 .9933 L
.58545 .9933 L
F
.58802 .98849 m
.59058 .98849 L
.59058 .9933 L
.58802 .9933 L
F
.59314 .98849 m
.59571 .98849 L
.59571 .9933 L
.59314 .9933 L
F
.54443 .9933 m
.54699 .9933 L
.54699 .99811 L
.54443 .99811 L
F
.59571 .98849 m
.59827 .98849 L
.59827 .9933 L
.59571 .9933 L
F
.54186 .9933 m
.54443 .9933 L
.54443 .99811 L
.54186 .99811 L
F
.58802 .98368 m
.59058 .98368 L
.59058 .98849 L
.58802 .98849 L
F
.59058 .98368 m
.59314 .98368 L
.59314 .98849 L
.59058 .98849 L
F
.57007 .98368 m
.57263 .98368 L
.57263 .98849 L
.57007 .98849 L
F
.57263 .98368 m
.5752 .98368 L
.5752 .98849 L
.57263 .98849 L
F
.58545 .98368 m
.58802 .98368 L
.58802 .98849 L
.58545 .98849 L
F
.59058 .98849 m
.59314 .98849 L
.59314 .9933 L
.59058 .9933 L
F
.58289 .98368 m
.58545 .98368 L
.58545 .98849 L
.58289 .98849 L
F
.58802 1.02216 m
.59058 1.02216 L
.59058 1.02697 L
.58802 1.02697 L
F
.59058 1.02216 m
.59314 1.02216 L
.59314 1.02697 L
.59058 1.02697 L
F
.57007 1.02216 m
.57263 1.02216 L
.57263 1.02697 L
.57007 1.02697 L
F
.57263 1.02216 m
.5752 1.02216 L
.5752 1.02697 L
.57263 1.02697 L
F
.58802 1.02697 m
.59058 1.02697 L
.59058 1.03178 L
.58802 1.03178 L
F
.58545 1.02697 m
.58802 1.02697 L
.58802 1.03178 L
.58545 1.03178 L
F
.58289 1.02216 m
.58545 1.02216 L
.58545 1.02697 L
.58289 1.02697 L
F
.58545 1.02216 m
.58802 1.02216 L
.58802 1.02697 L
.58545 1.02697 L
F
.54699 1.02697 m
.54955 1.02697 L
.54955 1.03178 L
.54699 1.03178 L
F
.54955 1.02697 m
.55212 1.02697 L
.55212 1.03178 L
.54955 1.03178 L
F
.55212 1.02697 m
.55468 1.02697 L
.55468 1.03178 L
.55212 1.03178 L
F
.55468 1.02697 m
.55725 1.02697 L
.55725 1.03178 L
.55468 1.03178 L
F
.5752 1.02216 m
.57776 1.02216 L
.57776 1.02697 L
.5752 1.02697 L
F
.58032 1.02216 m
.58289 1.02216 L
.58289 1.02697 L
.58032 1.02697 L
F
.57776 1.02216 m
.58032 1.02216 L
.58032 1.02697 L
.57776 1.02697 L
F
.5675 1.02697 m
.57007 1.02697 L
.57007 1.03178 L
.5675 1.03178 L
F
.55725 1.02697 m
.55981 1.02697 L
.55981 1.03178 L
.55725 1.03178 L
F
.56494 1.02697 m
.5675 1.02697 L
.5675 1.03178 L
.56494 1.03178 L
F
.56238 1.02697 m
.56494 1.02697 L
.56494 1.03178 L
.56238 1.03178 L
F
.5752 1.02697 m
.57776 1.02697 L
.57776 1.03178 L
.5752 1.03178 L
F
.57776 1.02697 m
.58032 1.02697 L
.58032 1.03178 L
.57776 1.03178 L
F
.55981 1.02697 m
.56238 1.02697 L
.56238 1.03178 L
.55981 1.03178 L
F
.58289 1.02697 m
.58545 1.02697 L
.58545 1.03178 L
.58289 1.03178 L
F
.59922 .84418 m
.59922 .849 L
.59827 .849 L
.59827 .84418 L
F
.59058 1.02697 m
.59314 1.02697 L
.59314 1.03178 L
.59058 1.03178 L
F
.58032 1.02697 m
.58289 1.02697 L
.58289 1.03178 L
.58032 1.03178 L
F
.59314 1.02697 m
.59571 1.02697 L
.59571 1.03178 L
.59314 1.03178 L
F
.59571 1.02697 m
.59827 1.02697 L
.59827 1.03178 L
.59571 1.03178 L
F
.59922 .849 m
.59922 .85381 L
.59827 .85381 L
.59827 .849 L
F
.57776 .93077 m
.58032 .93077 L
.58032 .93558 L
.57776 .93558 L
F
.58032 .93077 m
.58289 .93077 L
.58289 .93558 L
.58032 .93558 L
F
.55981 .93077 m
.56238 .93077 L
.56238 .93558 L
.55981 .93558 L
F
.56238 .93077 m
.56494 .93077 L
.56494 .93558 L
.56238 .93558 L
F
.57776 .93558 m
.58032 .93558 L
.58032 .94039 L
.57776 .94039 L
F
.5752 .93558 m
.57776 .93558 L
.57776 .94039 L
.5752 .94039 L
F
.57263 .93077 m
.5752 .93077 L
.5752 .93558 L
.57263 .93558 L
F
.5752 .93077 m
.57776 .93077 L
.57776 .93558 L
.5752 .93558 L
F
.59571 .93077 m
.59827 .93077 L
.59827 .93558 L
.59571 .93558 L
F
.5393 .93558 m
.54186 .93558 L
.54186 .94039 L
.5393 .94039 L
F
.54186 .93558 m
.54443 .93558 L
.54443 .94039 L
.54186 .94039 L
F
.54443 .93558 m
.54699 .93558 L
.54699 .94039 L
.54443 .94039 L
F
.56494 .93077 m
.5675 .93077 L
.5675 .93558 L
.56494 .93558 L
F
.57007 .93077 m
.57263 .93077 L
.57263 .93558 L
.57007 .93558 L
F
.5675 .93077 m
.57007 .93077 L
.57007 .93558 L
.5675 .93558 L
F
.55725 .93558 m
.55981 .93558 L
.55981 .94039 L
.55725 .94039 L
F
.54699 .93558 m
.54955 .93558 L
.54955 .94039 L
.54699 .94039 L
F
.55468 .93558 m
.55725 .93558 L
.55725 .94039 L
.55468 .94039 L
F
.55212 .93558 m
.55468 .93558 L
.55468 .94039 L
.55212 .94039 L
F
.5675 .93558 m
.57007 .93558 L
.57007 .94039 L
.5675 .94039 L
F
.55981 .93558 m
.56238 .93558 L
.56238 .94039 L
.55981 .94039 L
F
.54955 .93558 m
.55212 .93558 L
.55212 .94039 L
.54955 .94039 L
F
.56494 .93558 m
.5675 .93558 L
.5675 .94039 L
.56494 .94039 L
F
.57263 .93558 m
.5752 .93558 L
.5752 .94039 L
.57263 .94039 L
F
.58802 .93558 m
.59058 .93558 L
.59058 .94039 L
.58802 .94039 L
F
.58032 .93558 m
.58289 .93558 L
.58289 .94039 L
.58032 .94039 L
F
.57007 .93558 m
.57263 .93558 L
.57263 .94039 L
.57007 .94039 L
F
.58289 .93558 m
.58545 .93558 L
.58545 .94039 L
.58289 .94039 L
F
.58545 .93558 m
.58802 .93558 L
.58802 .94039 L
.58545 .94039 L
F
.59058 .93558 m
.59314 .93558 L
.59314 .94039 L
.59058 .94039 L
F
.54699 .9452 m
.54955 .9452 L
.54955 .95001 L
.54699 .95001 L
F
.54955 .9452 m
.55212 .9452 L
.55212 .95001 L
.54955 .95001 L
F
.55212 .9452 m
.55468 .9452 L
.55468 .95001 L
.55212 .95001 L
F
.55725 .9452 m
.55981 .9452 L
.55981 .95001 L
.55725 .95001 L
F
.55981 .9452 m
.56238 .9452 L
.56238 .95001 L
.55981 .95001 L
F
.56238 .9452 m
.56494 .9452 L
.56494 .95001 L
.56238 .95001 L
F
.56494 .9452 m
.5675 .9452 L
.5675 .95001 L
.56494 .95001 L
F
.5675 .9452 m
.57007 .9452 L
.57007 .95001 L
.5675 .95001 L
F
.55212 .94039 m
.55468 .94039 L
.55468 .9452 L
.55212 .9452 L
F
.55468 .94039 m
.55725 .94039 L
.55725 .9452 L
.55468 .9452 L
F
.59314 .93558 m
.59571 .93558 L
.59571 .94039 L
.59314 .94039 L
F
.59571 .93558 m
.59827 .93558 L
.59827 .94039 L
.59571 .94039 L
F
.54955 .94039 m
.55212 .94039 L
.55212 .9452 L
.54955 .9452 L
F
.55468 .9452 m
.55725 .9452 L
.55725 .95001 L
.55468 .95001 L
F
.54699 .94039 m
.54955 .94039 L
.54955 .9452 L
.54699 .9452 L
F
.59058 .93077 m
.59314 .93077 L
.59314 .93558 L
.59058 .93558 L
F
.58802 .93077 m
.59058 .93077 L
.59058 .93558 L
.58802 .93558 L
F
.59571 .94039 m
.59827 .94039 L
.59827 .9452 L
.59571 .9452 L
F
.5393 .9452 m
.54186 .9452 L
.54186 .95001 L
.5393 .95001 L
F
.58289 .93077 m
.58545 .93077 L
.58545 .93558 L
.58289 .93558 L
F
.58545 .93077 m
.58802 .93077 L
.58802 .93558 L
.58545 .93558 L
F
.59314 .93077 m
.59571 .93077 L
.59571 .93558 L
.59314 .93558 L
F
.59314 .94039 m
.59571 .94039 L
.59571 .9452 L
.59314 .9452 L
F
.58289 .94039 m
.58545 .94039 L
.58545 .9452 L
.58289 .9452 L
F
.59058 .94039 m
.59314 .94039 L
.59314 .9452 L
.59058 .9452 L
F
.58802 .94039 m
.59058 .94039 L
.59058 .9452 L
.58802 .9452 L
F
.54186 .9452 m
.54443 .9452 L
.54443 .95001 L
.54186 .95001 L
F
.54443 .9452 m
.54699 .9452 L
.54699 .95001 L
.54443 .95001 L
F
.58545 .94039 m
.58802 .94039 L
.58802 .9452 L
.58545 .9452 L
F
.58802 1.00773 m
.59058 1.00773 L
.59058 1.01254 L
.58802 1.01254 L
F
.58545 1.00773 m
.58802 1.00773 L
.58802 1.01254 L
.58545 1.01254 L
F
.59571 1.01735 m
.59827 1.01735 L
.59827 1.02216 L
.59571 1.02216 L
F
.5393 1.02216 m
.54186 1.02216 L
.54186 1.02697 L
.5393 1.02697 L
F
.58289 1.00773 m
.58545 1.00773 L
.58545 1.01254 L
.58289 1.01254 L
F
.54186 1.01254 m
.54443 1.01254 L
.54443 1.01735 L
.54186 1.01735 L
F
.59058 1.00773 m
.59314 1.00773 L
.59314 1.01254 L
.59058 1.01254 L
F
.58032 1.00773 m
.58289 1.00773 L
.58289 1.01254 L
.58032 1.01254 L
F
.59314 1.01735 m
.59571 1.01735 L
.59571 1.02216 L
.59314 1.02216 L
F
.58289 1.01735 m
.58545 1.01735 L
.58545 1.02216 L
.58289 1.02216 L
F
.59058 1.01735 m
.59314 1.01735 L
.59314 1.02216 L
.59058 1.02216 L
F
.58802 1.01735 m
.59058 1.01735 L
.59058 1.02216 L
.58802 1.02216 L
F
.54186 1.02216 m
.54443 1.02216 L
.54443 1.02697 L
.54186 1.02697 L
F
.54443 1.02216 m
.54699 1.02216 L
.54699 1.02697 L
.54443 1.02697 L
F
.58545 1.01735 m
.58802 1.01735 L
.58802 1.02216 L
.58545 1.02216 L
F
.5752 1.00773 m
.57776 1.00773 L
.57776 1.01254 L
.5752 1.01254 L
F
.57776 1.00773 m
.58032 1.00773 L
.58032 1.01254 L
.57776 1.01254 L
F
.55725 1.00773 m
.55981 1.00773 L
.55981 1.01254 L
.55725 1.01254 L
F
.55981 1.00773 m
.56238 1.00773 L
.56238 1.01254 L
.55981 1.01254 L
F
.57263 1.00773 m
.5752 1.00773 L
.5752 1.01254 L
.57263 1.01254 L
F
.57776 1.01254 m
.58032 1.01254 L
.58032 1.01735 L
.57776 1.01735 L
F
.57007 1.00773 m
.57263 1.00773 L
.57263 1.01254 L
.57007 1.01254 L
F
.59314 1.00773 m
.59571 1.00773 L
.59571 1.01254 L
.59314 1.01254 L
F
.59571 1.00773 m
.59827 1.00773 L
.59827 1.01254 L
.59571 1.01254 L
F
.5393 1.01254 m
.54186 1.01254 L
.54186 1.01735 L
.5393 1.01735 L
F
.54443 1.01254 m
.54699 1.01254 L
.54699 1.01735 L
.54443 1.01735 L
F
.56238 1.00773 m
.56494 1.00773 L
.56494 1.01254 L
.56238 1.01254 L
F
.5675 1.00773 m
.57007 1.00773 L
.57007 1.01254 L
.5675 1.01254 L
F
.56494 1.00773 m
.5675 1.00773 L
.5675 1.01254 L
.56494 1.01254 L
F
.57007 1.01735 m
.57263 1.01735 L
.57263 1.02216 L
.57007 1.02216 L
F
.57263 1.01735 m
.5752 1.01735 L
.5752 1.02216 L
.57263 1.02216 L
F
.5752 1.01735 m
.57776 1.01735 L
.57776 1.02216 L
.5752 1.02216 L
F
.58032 1.01735 m
.58289 1.01735 L
.58289 1.02216 L
.58032 1.02216 L
F
.54443 1.01735 m
.54699 1.01735 L
.54699 1.02216 L
.54443 1.02216 L
F
.59314 1.01254 m
.59571 1.01254 L
.59571 1.01735 L
.59314 1.01735 L
F
.54186 1.01735 m
.54443 1.01735 L
.54443 1.02216 L
.54186 1.02216 L
F
.5393 1.01735 m
.54186 1.01735 L
.54186 1.02216 L
.5393 1.02216 L
F
.56238 1.01735 m
.56494 1.01735 L
.56494 1.02216 L
.56238 1.02216 L
F
.5675 1.01735 m
.57007 1.01735 L
.57007 1.02216 L
.5675 1.02216 L
F
.56238 .93558 m
.56494 .93558 L
.56494 .94039 L
.56238 .94039 L
F
.56494 1.01735 m
.5675 1.01735 L
.5675 1.02216 L
.56494 1.02216 L
F
.55981 1.01735 m
.56238 1.01735 L
.56238 1.02216 L
.55981 1.02216 L
F
.57776 1.01735 m
.58032 1.01735 L
.58032 1.02216 L
.57776 1.02216 L
F
.55725 1.01735 m
.55981 1.01735 L
.55981 1.02216 L
.55725 1.02216 L
F
.54955 1.02216 m
.55212 1.02216 L
.55212 1.02697 L
.54955 1.02697 L
F
.56494 1.02216 m
.5675 1.02216 L
.5675 1.02697 L
.56494 1.02697 L
F
.55725 1.02216 m
.55981 1.02216 L
.55981 1.02697 L
.55725 1.02697 L
F
.54699 1.02216 m
.54955 1.02216 L
.54955 1.02697 L
.54699 1.02697 L
F
.55981 1.02216 m
.56238 1.02216 L
.56238 1.02697 L
.55981 1.02697 L
F
.56238 1.02216 m
.56494 1.02216 L
.56494 1.02697 L
.56238 1.02697 L
F
.5675 1.02216 m
.57007 1.02216 L
.57007 1.02697 L
.5675 1.02697 L
F
.55468 1.01735 m
.55725 1.01735 L
.55725 1.02216 L
.55468 1.02216 L
F
.54699 1.01735 m
.54955 1.01735 L
.54955 1.02216 L
.54699 1.02216 L
F
.59571 1.01254 m
.59827 1.01254 L
.59827 1.01735 L
.59571 1.01735 L
F
.55212 1.01735 m
.55468 1.01735 L
.55468 1.02216 L
.55212 1.02216 L
F
.55468 1.02216 m
.55725 1.02216 L
.55725 1.02697 L
.55468 1.02697 L
F
.55212 1.02216 m
.55468 1.02216 L
.55468 1.02697 L
.55212 1.02697 L
F
.54955 1.01735 m
.55212 1.01735 L
.55212 1.02216 L
.54955 1.02216 L
F
.5 g
.63417 .95482 m
.63673 .95482 L
.63673 .95963 L
.63417 .95963 L
F
.62904 .95482 m
.63161 .95482 L
.63161 .95963 L
.62904 .95963 L
F
.63161 .95482 m
.63417 .95482 L
.63417 .95963 L
.63161 .95963 L
F
.61879 .88748 m
.62135 .88748 L
.62135 .89229 L
.61879 .89229 L
F
.62135 .88748 m
.62391 .88748 L
.62391 .89229 L
.62135 .89229 L
F
.63161 .9452 m
.63417 .9452 L
.63417 .95001 L
.63161 .95001 L
F
.62648 .9452 m
.62904 .9452 L
.62904 .95001 L
.62648 .95001 L
F
.62904 .9452 m
.63161 .9452 L
.63161 .95001 L
.62904 .95001 L
F
.63417 .9452 m
.63673 .9452 L
.63673 .95001 L
.63417 .95001 L
F
.63673 .9452 m
.6393 .9452 L
.6393 .95001 L
.63673 .95001 L
F
.60084 .88748 m
.6034 .88748 L
.6034 .89229 L
.60084 .89229 L
F
.59922 .88748 m
.60084 .88748 L
.60084 .89229 L
.59922 .89229 L
F
.62391 .95482 m
.62648 .95482 L
.62648 .95963 L
.62391 .95963 L
F
.63161 .94039 m
.63417 .94039 L
.63417 .9452 L
.63161 .9452 L
F
.6034 .88748 m
.60597 .88748 L
.60597 .89229 L
.6034 .89229 L
F
.61879 .95482 m
.62135 .95482 L
.62135 .95963 L
.61879 .95963 L
F
.62135 .95482 m
.62391 .95482 L
.62391 .95963 L
.62135 .95963 L
F
.63673 .95482 m
.6393 .95482 L
.6393 .95963 L
.63673 .95963 L
F
.62648 .95482 m
.62904 .95482 L
.62904 .95963 L
.62648 .95963 L
F
.61622 .95482 m
.61879 .95482 L
.61879 .95963 L
.61622 .95963 L
F
.62135 .9452 m
.62391 .9452 L
.62391 .95001 L
.62135 .95001 L
F
.62391 .9452 m
.62648 .9452 L
.62648 .95001 L
.62391 .95001 L
F
.61109 .9452 m
.61366 .9452 L
.61366 .95001 L
.61109 .95001 L
F
.60084 .9452 m
.6034 .9452 L
.6034 .95001 L
.60084 .95001 L
F
.61879 .9452 m
.62135 .9452 L
.62135 .95001 L
.61879 .95001 L
F
.60853 .95001 m
.61109 .95001 L
.61109 .95482 L
.60853 .95482 L
F
.60597 .95001 m
.60853 .95001 L
.60853 .95482 L
.60597 .95482 L
F
.59922 .95001 m
.60084 .95001 L
.60084 .95482 L
.59922 .95482 L
F
.61622 .9452 m
.61879 .9452 L
.61879 .95001 L
.61622 .95001 L
F
.61366 .9452 m
.61622 .9452 L
.61622 .95001 L
.61366 .95001 L
F
.63161 .89229 m
.63417 .89229 L
.63417 .8971 L
.63161 .8971 L
F
.63673 .89229 m
.6393 .89229 L
.6393 .8971 L
.63673 .8971 L
F
.63417 .89229 m
.63673 .89229 L
.63673 .8971 L
.63417 .8971 L
F
.62904 .89229 m
.63161 .89229 L
.63161 .8971 L
.62904 .8971 L
F
.62648 .89229 m
.62904 .89229 L
.62904 .8971 L
.62648 .8971 L
F
.59922 .9452 m
.60084 .9452 L
.60084 .95001 L
.59922 .95001 L
F
.6034 .9452 m
.60597 .9452 L
.60597 .95001 L
.6034 .95001 L
F
.60597 .9452 m
.60853 .9452 L
.60853 .95001 L
.60597 .95001 L
F
.60853 .9452 m
.61109 .9452 L
.61109 .95001 L
.60853 .95001 L
F
.60084 .95482 m
.6034 .95482 L
.6034 .95963 L
.60084 .95963 L
F
.59922 .95482 m
.60084 .95482 L
.60084 .95963 L
.59922 .95963 L
F
.62904 .88748 m
.63161 .88748 L
.63161 .89229 L
.62904 .89229 L
F
.63161 .98368 m
.63417 .98368 L
.63417 .98849 L
.63161 .98849 L
F
.62904 .98368 m
.63161 .98368 L
.63161 .98849 L
.62904 .98849 L
F
.61366 .95482 m
.61622 .95482 L
.61622 .95963 L
.61366 .95963 L
F
.61109 .95482 m
.61366 .95482 L
.61366 .95963 L
.61109 .95963 L
F
.6034 .95482 m
.60597 .95482 L
.60597 .95963 L
.6034 .95963 L
F
.62391 .88748 m
.62648 .88748 L
.62648 .89229 L
.62391 .89229 L
F
.62648 .88748 m
.62904 .88748 L
.62904 .89229 L
.62648 .89229 L
F
.61366 .98368 m
.61622 .98368 L
.61622 .98849 L
.61366 .98849 L
F
.62391 .98368 m
.62648 .98368 L
.62648 .98849 L
.62391 .98849 L
F
.61879 .98368 m
.62135 .98368 L
.62135 .98849 L
.61879 .98849 L
F
.61109 .93077 m
.61366 .93077 L
.61366 .93558 L
.61109 .93558 L
F
.62135 .98368 m
.62391 .98368 L
.62391 .98849 L
.62135 .98849 L
F
.63673 .98368 m
.6393 .98368 L
.6393 .98849 L
.63673 .98849 L
F
.62904 .93077 m
.63161 .93077 L
.63161 .93558 L
.62904 .93558 L
F
.62648 .98368 m
.62904 .98368 L
.62904 .98849 L
.62648 .98849 L
F
.61622 .98368 m
.61879 .98368 L
.61879 .98849 L
.61622 .98849 L
F
.63417 .98368 m
.63673 .98368 L
.63673 .98849 L
.63417 .98849 L
F
.59922 .8971 m
.60084 .8971 L
.60084 .90191 L
.59922 .90191 L
F
.63673 .94039 m
.6393 .94039 L
.6393 .9452 L
.63673 .9452 L
F
.60853 .88748 m
.61109 .88748 L
.61109 .89229 L
.60853 .89229 L
F
.63417 .88748 m
.63673 .88748 L
.63673 .89229 L
.63417 .89229 L
F
.63673 .95001 m
.6393 .95001 L
.6393 .95482 L
.63673 .95482 L
F
.61366 .88748 m
.61622 .88748 L
.61622 .89229 L
.61366 .89229 L
F
.61109 .88748 m
.61366 .88748 L
.61366 .89229 L
.61109 .89229 L
F
.63417 .94039 m
.63673 .94039 L
.63673 .9452 L
.63417 .9452 L
F
.60597 .88748 m
.60853 .88748 L
.60853 .89229 L
.60597 .89229 L
F
.61622 .88748 m
.61879 .88748 L
.61879 .89229 L
.61622 .89229 L
F
.63417 .95001 m
.63673 .95001 L
.63673 .95482 L
.63417 .95482 L
F
.63161 .95001 m
.63417 .95001 L
.63417 .95482 L
.63161 .95482 L
F
.62391 .95001 m
.62648 .95001 L
.62648 .95482 L
.62391 .95482 L
F
.60853 .95482 m
.61109 .95482 L
.61109 .95963 L
.60853 .95963 L
F
.60597 .95482 m
.60853 .95482 L
.60853 .95963 L
.60597 .95963 L
F
.63673 .88748 m
.6393 .88748 L
.6393 .89229 L
.63673 .89229 L
F
.63161 .88748 m
.63417 .88748 L
.63417 .89229 L
.63161 .89229 L
F
.62904 .95001 m
.63161 .95001 L
.63161 .95482 L
.62904 .95482 L
F
.62648 .95001 m
.62904 .95001 L
.62904 .95482 L
.62648 .95482 L
F
.63673 .96444 m
.6393 .96444 L
.6393 .96925 L
.63673 .96925 L
F
.63161 .96444 m
.63417 .96444 L
.63417 .96925 L
.63161 .96925 L
F
.63417 .96444 m
.63673 .96444 L
.63673 .96925 L
.63417 .96925 L
F
.6034 .88267 m
.60597 .88267 L
.60597 .88748 L
.6034 .88748 L
F
.60084 .88267 m
.6034 .88267 L
.6034 .88748 L
.60084 .88748 L
F
.59922 .88267 m
.60084 .88267 L
.60084 .88748 L
.59922 .88748 L
F
.62904 .8971 m
.63161 .8971 L
.63161 .90191 L
.62904 .90191 L
F
.62391 .8971 m
.62648 .8971 L
.62648 .90191 L
.62391 .90191 L
F
.62648 .8971 m
.62904 .8971 L
.62904 .90191 L
.62648 .90191 L
F
.63161 .8971 m
.63417 .8971 L
.63417 .90191 L
.63161 .90191 L
F
.63673 .93558 m
.6393 .93558 L
.6393 .94039 L
.63673 .94039 L
F
.63417 .8971 m
.63673 .8971 L
.63673 .90191 L
.63417 .90191 L
F
.63673 .8971 m
.6393 .8971 L
.6393 .90191 L
.63673 .90191 L
F
.63417 .93558 m
.63673 .93558 L
.63673 .94039 L
.63417 .94039 L
F
.63161 .93558 m
.63417 .93558 L
.63417 .94039 L
.63161 .94039 L
F
.59922 .96925 m
.60084 .96925 L
.60084 .97406 L
.59922 .97406 L
F
.62904 .93558 m
.63161 .93558 L
.63161 .94039 L
.62904 .94039 L
F
.62648 .93558 m
.62904 .93558 L
.62904 .94039 L
.62648 .94039 L
F
.60084 .96925 m
.6034 .96925 L
.6034 .97406 L
.60084 .97406 L
F
.6034 .96925 m
.60597 .96925 L
.60597 .97406 L
.6034 .97406 L
F
.61879 .94039 m
.62135 .94039 L
.62135 .9452 L
.61879 .9452 L
F
.62135 .94039 m
.62391 .94039 L
.62391 .9452 L
.62135 .9452 L
F
.62904 .94039 m
.63161 .94039 L
.63161 .9452 L
.62904 .9452 L
F
.61622 .95963 m
.61879 .95963 L
.61879 .96444 L
.61622 .96444 L
F
.61366 .95963 m
.61622 .95963 L
.61622 .96444 L
.61366 .96444 L
F
.6034 .94039 m
.60597 .94039 L
.60597 .9452 L
.6034 .9452 L
F
.60853 .94039 m
.61109 .94039 L
.61109 .9452 L
.60853 .9452 L
F
.60597 .94039 m
.60853 .94039 L
.60853 .9452 L
.60597 .9452 L
F
.62391 .94039 m
.62648 .94039 L
.62648 .9452 L
.62391 .9452 L
F
.62648 .94039 m
.62904 .94039 L
.62904 .9452 L
.62648 .9452 L
F
.59922 .95963 m
.60084 .95963 L
.60084 .96444 L
.59922 .96444 L
F
.60853 .95963 m
.61109 .95963 L
.61109 .96444 L
.60853 .96444 L
F
.6034 .95963 m
.60597 .95963 L
.60597 .96444 L
.6034 .96444 L
F
.62135 .8971 m
.62391 .8971 L
.62391 .90191 L
.62135 .90191 L
F
.60597 .95963 m
.60853 .95963 L
.60853 .96444 L
.60597 .96444 L
F
.62135 .95963 m
.62391 .95963 L
.62391 .96444 L
.62135 .96444 L
F
.61109 .95963 m
.61366 .95963 L
.61366 .96444 L
.61109 .96444 L
F
.60084 .95963 m
.6034 .95963 L
.6034 .96444 L
.60084 .96444 L
F
.61879 .95963 m
.62135 .95963 L
.62135 .96444 L
.61879 .96444 L
F
.59922 .89229 m
.60084 .89229 L
.60084 .8971 L
.59922 .8971 L
F
.6034 .89229 m
.60597 .89229 L
.60597 .8971 L
.6034 .8971 L
F
.60084 .89229 m
.6034 .89229 L
.6034 .8971 L
.60084 .8971 L
F
.61879 .89229 m
.62135 .89229 L
.62135 .8971 L
.61879 .8971 L
F
.62135 .89229 m
.62391 .89229 L
.62391 .8971 L
.62135 .8971 L
F
.61366 .96444 m
.61622 .96444 L
.61622 .96925 L
.61366 .96925 L
F
.60853 .89229 m
.61109 .89229 L
.61109 .8971 L
.60853 .8971 L
F
.61622 .96444 m
.61879 .96444 L
.61879 .96925 L
.61622 .96925 L
F
.61109 .89229 m
.61366 .89229 L
.61366 .8971 L
.61109 .8971 L
F
.60597 .89229 m
.60853 .89229 L
.60853 .8971 L
.60597 .8971 L
F
.62135 .95001 m
.62391 .95001 L
.62391 .95482 L
.62135 .95482 L
F
.61879 .95001 m
.62135 .95001 L
.62135 .95482 L
.61879 .95482 L
F
.61109 .95001 m
.61366 .95001 L
.61366 .95482 L
.61109 .95482 L
F
.6034 .95001 m
.60597 .95001 L
.60597 .95482 L
.6034 .95482 L
F
.60084 .95001 m
.6034 .95001 L
.6034 .95482 L
.60084 .95482 L
F
.61366 .89229 m
.61622 .89229 L
.61622 .8971 L
.61366 .8971 L
F
.61622 .89229 m
.61879 .89229 L
.61879 .8971 L
.61622 .8971 L
F
.62391 .89229 m
.62648 .89229 L
.62648 .8971 L
.62391 .8971 L
F
.61622 .95001 m
.61879 .95001 L
.61879 .95482 L
.61622 .95482 L
F
.61366 .95001 m
.61622 .95001 L
.61622 .95482 L
.61366 .95482 L
F
.62135 .88267 m
.62391 .88267 L
.62391 .88748 L
.62135 .88748 L
F
.61879 .88267 m
.62135 .88267 L
.62135 .88748 L
.61879 .88748 L
F
.59922 .96444 m
.60084 .96444 L
.60084 .96925 L
.59922 .96925 L
F
.61366 .88267 m
.61622 .88267 L
.61622 .88748 L
.61366 .88748 L
F
.62391 .88267 m
.62648 .88267 L
.62648 .88748 L
.62391 .88748 L
F
.60597 .88267 m
.60853 .88267 L
.60853 .88748 L
.60597 .88748 L
F
.61109 .88267 m
.61366 .88267 L
.61366 .88748 L
.61109 .88748 L
F
.60853 .88267 m
.61109 .88267 L
.61109 .88748 L
.60853 .88748 L
F
.6034 .96444 m
.60597 .96444 L
.60597 .96925 L
.6034 .96925 L
F
.60084 .96444 m
.6034 .96444 L
.6034 .96925 L
.60084 .96925 L
F
.62648 .96444 m
.62904 .96444 L
.62904 .96925 L
.62648 .96925 L
F
.60853 .96444 m
.61109 .96444 L
.61109 .96925 L
.60853 .96925 L
F
.62904 .96444 m
.63161 .96444 L
.63161 .96925 L
.62904 .96925 L
F
.60597 .96444 m
.60853 .96444 L
.60853 .96925 L
.60597 .96925 L
F
.61109 .96444 m
.61366 .96444 L
.61366 .96925 L
.61109 .96925 L
F
.62135 .96444 m
.62391 .96444 L
.62391 .96925 L
.62135 .96925 L
F
.61622 .88267 m
.61879 .88267 L
.61879 .88748 L
.61622 .88748 L
F
.61879 .96444 m
.62135 .96444 L
.62135 .96925 L
.61879 .96925 L
F
.62391 .96444 m
.62648 .96444 L
.62648 .96925 L
.62391 .96925 L
F
.61366 .96925 m
.61622 .96925 L
.61622 .97406 L
.61366 .97406 L
F
.63417 .87786 m
.63673 .87786 L
.63673 .88267 L
.63417 .88267 L
F
.61622 .96925 m
.61879 .96925 L
.61879 .97406 L
.61622 .97406 L
F
.63673 .87786 m
.6393 .87786 L
.6393 .88267 L
.63673 .88267 L
F
.63161 .87786 m
.63417 .87786 L
.63417 .88267 L
.63161 .88267 L
F
.62648 .96925 m
.62904 .96925 L
.62904 .97406 L
.62648 .97406 L
F
.60853 .96925 m
.61109 .96925 L
.61109 .97406 L
.60853 .97406 L
F
.63161 .96925 m
.63417 .96925 L
.63417 .97406 L
.63161 .97406 L
F
.60597 .96925 m
.60853 .96925 L
.60853 .97406 L
.60597 .97406 L
F
.61109 .96925 m
.61366 .96925 L
.61366 .97406 L
.61109 .97406 L
F
.62135 .93558 m
.62391 .93558 L
.62391 .94039 L
.62135 .94039 L
F
.62135 .97887 m
.62391 .97887 L
.62391 .98368 L
.62135 .98368 L
F
.61879 .93558 m
.62135 .93558 L
.62135 .94039 L
.61879 .94039 L
F
.61879 .97887 m
.62135 .97887 L
.62135 .98368 L
.61879 .98368 L
F
.62391 .97887 m
.62648 .97887 L
.62648 .98368 L
.62391 .98368 L
F
.62904 .87786 m
.63161 .87786 L
.63161 .88267 L
.62904 .88267 L
F
.63673 .96925 m
.6393 .96925 L
.6393 .97406 L
.63673 .97406 L
F
.62648 .87786 m
.62904 .87786 L
.62904 .88267 L
.62648 .88267 L
F
.63417 .96925 m
.63673 .96925 L
.63673 .97406 L
.63417 .97406 L
F
.62391 .93558 m
.62648 .93558 L
.62648 .94039 L
.62391 .94039 L
F
.60597 .97406 m
.60853 .97406 L
.60853 .97887 L
.60597 .97887 L
F
.60084 .97406 m
.6034 .97406 L
.6034 .97887 L
.60084 .97887 L
F
.6034 .97406 m
.60597 .97406 L
.60597 .97887 L
.6034 .97887 L
F
.60853 .97406 m
.61109 .97406 L
.61109 .97887 L
.60853 .97887 L
F
.61109 .97406 m
.61366 .97406 L
.61366 .97887 L
.61109 .97887 L
F
.61879 .87786 m
.62135 .87786 L
.62135 .88267 L
.61879 .88267 L
F
.62391 .87786 m
.62648 .87786 L
.62648 .88267 L
.62391 .88267 L
F
.62135 .87786 m
.62391 .87786 L
.62391 .88267 L
.62135 .88267 L
F
.61622 .87786 m
.61879 .87786 L
.61879 .88267 L
.61622 .88267 L
F
.61366 .87786 m
.61622 .87786 L
.61622 .88267 L
.61366 .88267 L
F
.62135 .96925 m
.62391 .96925 L
.62391 .97406 L
.62135 .97406 L
F
.62391 .96925 m
.62648 .96925 L
.62648 .97406 L
.62391 .97406 L
F
.60853 .93558 m
.61109 .93558 L
.61109 .94039 L
.60853 .94039 L
F
.62904 .96925 m
.63161 .96925 L
.63161 .97406 L
.62904 .97406 L
F
.61879 .96925 m
.62135 .96925 L
.62135 .97406 L
.61879 .97406 L
F
.61109 .93558 m
.61366 .93558 L
.61366 .94039 L
.61109 .94039 L
F
.61366 .93558 m
.61622 .93558 L
.61622 .94039 L
.61366 .94039 L
F
.59922 .97406 m
.60084 .97406 L
.60084 .97887 L
.59922 .97887 L
F
.61622 .93558 m
.61879 .93558 L
.61879 .94039 L
.61622 .94039 L
F
.61879 .97406 m
.62135 .97406 L
.62135 .97887 L
.61879 .97887 L
F
.61366 .97406 m
.61622 .97406 L
.61622 .97887 L
.61366 .97887 L
F
.61622 .97406 m
.61879 .97406 L
.61879 .97887 L
.61622 .97887 L
F
.62135 .97406 m
.62391 .97406 L
.62391 .97887 L
.62135 .97887 L
F
.62391 .97406 m
.62648 .97406 L
.62648 .97887 L
.62391 .97887 L
F
.62648 .97406 m
.62904 .97406 L
.62904 .97887 L
.62648 .97887 L
F
.63673 .97406 m
.6393 .97406 L
.6393 .97887 L
.63673 .97887 L
F
.63161 .97406 m
.63417 .97406 L
.63417 .97887 L
.63161 .97887 L
F
.63417 .97406 m
.63673 .97406 L
.63673 .97887 L
.63417 .97887 L
F
.60597 .93558 m
.60853 .93558 L
.60853 .94039 L
.60597 .94039 L
F
.59922 .93558 m
.60084 .93558 L
.60084 .94039 L
.59922 .94039 L
F
.6034 .93558 m
.60597 .93558 L
.60597 .94039 L
.6034 .94039 L
F
.60084 .93558 m
.6034 .93558 L
.6034 .94039 L
.60084 .94039 L
F
.63673 .87305 m
.6393 .87305 L
.6393 .87786 L
.63673 .87786 L
F
.63417 .87305 m
.63673 .87305 L
.63673 .87786 L
.63417 .87786 L
F
.59922 .97887 m
.60084 .97887 L
.60084 .98368 L
.59922 .98368 L
F
.63161 .87305 m
.63417 .87305 L
.63417 .87786 L
.63161 .87786 L
F
.62904 .87305 m
.63161 .87305 L
.63161 .87786 L
.62904 .87786 L
F
.60084 .97887 m
.6034 .97887 L
.6034 .98368 L
.60084 .98368 L
F
.6034 .97887 m
.60597 .97887 L
.60597 .98368 L
.6034 .98368 L
F
.61622 .97887 m
.61879 .97887 L
.61879 .98368 L
.61622 .98368 L
F
.61366 .97887 m
.61622 .97887 L
.61622 .98368 L
.61366 .98368 L
F
.60597 .97887 m
.60853 .97887 L
.60853 .98368 L
.60597 .98368 L
F
.61366 .87305 m
.61622 .87305 L
.61622 .87786 L
.61366 .87786 L
F
.61622 .87305 m
.61879 .87305 L
.61879 .87786 L
.61622 .87786 L
F
.63161 .97887 m
.63417 .97887 L
.63417 .98368 L
.63161 .98368 L
F
.62648 .97887 m
.62904 .97887 L
.62904 .98368 L
.62648 .98368 L
F
.62904 .97887 m
.63161 .97887 L
.63161 .98368 L
.62904 .98368 L
F
.61109 .97887 m
.61366 .97887 L
.61366 .98368 L
.61109 .98368 L
F
.60853 .97887 m
.61109 .97887 L
.61109 .98368 L
.60853 .98368 L
F
.63417 .97887 m
.63673 .97887 L
.63673 .98368 L
.63417 .98368 L
F
.62135 .87305 m
.62391 .87305 L
.62391 .87786 L
.62135 .87786 L
F
.62648 .87305 m
.62904 .87305 L
.62904 .87786 L
.62648 .87786 L
F
.62904 .97406 m
.63161 .97406 L
.63161 .97887 L
.62904 .97887 L
F
.62391 .87305 m
.62648 .87305 L
.62648 .87786 L
.62391 .87786 L
F
.60853 .87305 m
.61109 .87305 L
.61109 .87786 L
.60853 .87786 L
F
.61879 .87305 m
.62135 .87305 L
.62135 .87786 L
.61879 .87786 L
F
.63673 .97887 m
.6393 .97887 L
.6393 .98368 L
.63673 .98368 L
F
.61109 .87305 m
.61366 .87305 L
.61366 .87786 L
.61109 .87786 L
F
.63673 .93077 m
.6393 .93077 L
.6393 .93558 L
.63673 .93558 L
F
.63417 .93077 m
.63673 .93077 L
.63673 .93558 L
.63417 .93558 L
F
.60853 .98368 m
.61109 .98368 L
.61109 .98849 L
.60853 .98849 L
F
.59922 .98368 m
.60084 .98368 L
.60084 .98849 L
.59922 .98849 L
F
.61622 .90191 m
.61879 .90191 L
.61879 .90672 L
.61622 .90672 L
F
.6034 .98368 m
.60597 .98368 L
.60597 .98849 L
.6034 .98849 L
F
.60597 .98368 m
.60853 .98368 L
.60853 .98849 L
.60597 .98849 L
F
.59922 .87305 m
.60084 .87305 L
.60084 .87786 L
.59922 .87786 L
F
.61109 .98368 m
.61366 .98368 L
.61366 .98849 L
.61109 .98849 L
F
.60084 .98368 m
.6034 .98368 L
.6034 .98849 L
.60084 .98849 L
F
.60597 .86824 m
.60853 .86824 L
.60853 .87305 L
.60597 .87305 L
F
.62391 .98849 m
.62648 .98849 L
.62648 .9933 L
.62391 .9933 L
F
.60084 .86824 m
.6034 .86824 L
.6034 .87305 L
.60084 .87305 L
F
.62135 .98849 m
.62391 .98849 L
.62391 .9933 L
.62135 .9933 L
F
.62648 .98849 m
.62904 .98849 L
.62904 .9933 L
.62648 .9933 L
F
.63673 .98849 m
.6393 .98849 L
.6393 .9933 L
.63673 .9933 L
F
.60853 .86824 m
.61109 .86824 L
.61109 .87305 L
.60853 .87305 L
F
.63161 .93077 m
.63417 .93077 L
.63417 .93558 L
.63161 .93558 L
F
.6034 .86824 m
.60597 .86824 L
.60597 .87305 L
.6034 .87305 L
F
.63417 .98849 m
.63673 .98849 L
.63673 .9933 L
.63417 .9933 L
F
.62648 .93077 m
.62904 .93077 L
.62904 .93558 L
.62648 .93558 L
F
.61622 .93077 m
.61879 .93077 L
.61879 .93558 L
.61622 .93558 L
F
.62135 .93077 m
.62391 .93077 L
.62391 .93558 L
.62135 .93558 L
F
.6034 .90191 m
.60597 .90191 L
.60597 .90672 L
.6034 .90672 L
F
.61879 .93077 m
.62135 .93077 L
.62135 .93558 L
.61879 .93558 L
F
.61366 .93077 m
.61622 .93077 L
.61622 .93558 L
.61366 .93558 L
F
.6034 .93077 m
.60597 .93077 L
.60597 .93558 L
.6034 .93558 L
F
.60853 .93077 m
.61109 .93077 L
.61109 .93558 L
.60853 .93558 L
F
.62391 .93077 m
.62648 .93077 L
.62648 .93558 L
.62391 .93558 L
F
.60597 .93077 m
.60853 .93077 L
.60853 .93558 L
.60597 .93558 L
F
.6034 .87305 m
.60597 .87305 L
.60597 .87786 L
.6034 .87786 L
F
.60084 .87305 m
.6034 .87305 L
.6034 .87786 L
.60084 .87786 L
F
.60853 .90191 m
.61109 .90191 L
.61109 .90672 L
.60853 .90672 L
F
.59922 .90191 m
.60084 .90191 L
.60084 .90672 L
.59922 .90672 L
F
.60597 .87305 m
.60853 .87305 L
.60853 .87786 L
.60597 .87786 L
F
.60084 .90191 m
.6034 .90191 L
.6034 .90672 L
.60084 .90672 L
F
.60597 .90191 m
.60853 .90191 L
.60853 .90672 L
.60597 .90672 L
F
.61366 .90191 m
.61622 .90191 L
.61622 .90672 L
.61366 .90672 L
F
.61109 .90191 m
.61366 .90191 L
.61366 .90672 L
.61109 .90672 L
F
.60853 .98849 m
.61109 .98849 L
.61109 .9933 L
.60853 .9933 L
F
.61879 .98849 m
.62135 .98849 L
.62135 .9933 L
.61879 .9933 L
F
.61366 .98849 m
.61622 .98849 L
.61622 .9933 L
.61366 .9933 L
F
.59922 .98849 m
.60084 .98849 L
.60084 .9933 L
.59922 .9933 L
F
.61622 .98849 m
.61879 .98849 L
.61879 .9933 L
.61622 .9933 L
F
.60084 .93077 m
.6034 .93077 L
.6034 .93558 L
.60084 .93558 L
F
.62904 .86824 m
.63161 .86824 L
.63161 .87305 L
.62904 .87305 L
F
.63417 .86824 m
.63673 .86824 L
.63673 .87305 L
.63417 .87305 L
F
.61109 .98849 m
.61366 .98849 L
.61366 .9933 L
.61109 .9933 L
F
.63161 .86824 m
.63417 .86824 L
.63417 .87305 L
.63161 .87305 L
F
.61109 .87786 m
.61366 .87786 L
.61366 .88267 L
.61109 .88267 L
F
.60084 .87786 m
.6034 .87786 L
.6034 .88267 L
.60084 .88267 L
F
.60597 .87786 m
.60853 .87786 L
.60853 .88267 L
.60597 .88267 L
F
.6034 .87786 m
.60597 .87786 L
.60597 .88267 L
.6034 .88267 L
F
.59922 .87786 m
.60084 .87786 L
.60084 .88267 L
.59922 .88267 L
F
.61109 .86824 m
.61366 .86824 L
.61366 .87305 L
.61109 .87305 L
F
.60597 .98849 m
.60853 .98849 L
.60853 .9933 L
.60597 .9933 L
F
.60084 .98849 m
.6034 .98849 L
.6034 .9933 L
.60084 .9933 L
F
.60853 .87786 m
.61109 .87786 L
.61109 .88267 L
.60853 .88267 L
F
.6034 .98849 m
.60597 .98849 L
.60597 .9933 L
.6034 .9933 L
F
.63417 .90191 m
.63673 .90191 L
.63673 .90672 L
.63417 .90672 L
F
.59922 .93077 m
.60084 .93077 L
.60084 .93558 L
.59922 .93558 L
F
.63673 .90191 m
.6393 .90191 L
.6393 .90672 L
.63673 .90672 L
F
.59922 .86824 m
.60084 .86824 L
.60084 .87305 L
.59922 .87305 L
F
.61879 .90191 m
.62135 .90191 L
.62135 .90672 L
.61879 .90672 L
F
.62904 .98849 m
.63161 .98849 L
.63161 .9933 L
.62904 .9933 L
F
.62904 .90191 m
.63161 .90191 L
.63161 .90672 L
.62904 .90672 L
F
.63161 .98849 m
.63417 .98849 L
.63417 .9933 L
.63161 .9933 L
F
.62648 .90191 m
.62904 .90191 L
.62904 .90672 L
.62648 .90672 L
F
.63161 .90191 m
.63417 .90191 L
.63417 .90672 L
.63161 .90672 L
F
.61622 .86824 m
.61879 .86824 L
.61879 .87305 L
.61622 .87305 L
F
.61366 .86824 m
.61622 .86824 L
.61622 .87305 L
.61366 .87305 L
F
.62648 .86824 m
.62904 .86824 L
.62904 .87305 L
.62648 .87305 L
F
.63673 .86824 m
.6393 .86824 L
.6393 .87305 L
.63673 .87305 L
F
.61879 .86824 m
.62135 .86824 L
.62135 .87305 L
.61879 .87305 L
F
.62135 .90191 m
.62391 .90191 L
.62391 .90672 L
.62135 .90672 L
F
.62391 .90191 m
.62648 .90191 L
.62648 .90672 L
.62391 .90672 L
F
.62135 .86824 m
.62391 .86824 L
.62391 .87305 L
.62135 .87305 L
F
.62391 .86824 m
.62648 .86824 L
.62648 .87305 L
.62391 .87305 L
F
.61879 .9933 m
.62135 .9933 L
.62135 .99811 L
.61879 .99811 L
F
.6034 .9933 m
.60597 .9933 L
.60597 .99811 L
.6034 .99811 L
F
.62135 .9933 m
.62391 .9933 L
.62391 .99811 L
.62135 .99811 L
F
.60853 .9933 m
.61109 .9933 L
.61109 .99811 L
.60853 .99811 L
F
.60084 .9933 m
.6034 .9933 L
.6034 .99811 L
.60084 .99811 L
F
.63673 .91153 m
.6393 .91153 L
.6393 .91634 L
.63673 .91634 L
F
.61366 .9933 m
.61622 .9933 L
.61622 .99811 L
.61366 .99811 L
F
.63417 .91153 m
.63673 .91153 L
.63673 .91634 L
.63417 .91634 L
F
.61109 .9933 m
.61366 .9933 L
.61366 .99811 L
.61109 .99811 L
F
.61622 .9933 m
.61879 .9933 L
.61879 .99811 L
.61622 .99811 L
F
.61879 .99811 m
.62135 .99811 L
.62135 1.00292 L
.61879 1.00292 L
F
.61622 .99811 m
.61879 .99811 L
.61879 1.00292 L
.61622 1.00292 L
F
.63417 .99811 m
.63673 .99811 L
.63673 1.00292 L
.63417 1.00292 L
F
.62135 .99811 m
.62391 .99811 L
.62391 1.00292 L
.62135 1.00292 L
F
.62391 .99811 m
.62648 .99811 L
.62648 1.00292 L
.62391 1.00292 L
F
.62904 .99811 m
.63161 .99811 L
.63161 1.00292 L
.62904 1.00292 L
F
.63161 .99811 m
.63417 .99811 L
.63417 1.00292 L
.63161 1.00292 L
F
.60597 .9933 m
.60853 .9933 L
.60853 .99811 L
.60597 .99811 L
F
.63673 .99811 m
.6393 .99811 L
.6393 1.00292 L
.63673 1.00292 L
F
.62648 .99811 m
.62904 .99811 L
.62904 1.00292 L
.62648 1.00292 L
F
.62135 .86343 m
.62391 .86343 L
.62391 .86824 L
.62135 .86824 L
F
.62391 .86343 m
.62648 .86343 L
.62648 .86824 L
.62391 .86824 L
F
.63161 .86343 m
.63417 .86343 L
.63417 .86824 L
.63161 .86824 L
F
.61366 .91153 m
.61622 .91153 L
.61622 .91634 L
.61366 .91634 L
F
.61109 .91153 m
.61366 .91153 L
.61366 .91634 L
.61109 .91634 L
F
.63161 .9933 m
.63417 .9933 L
.63417 .99811 L
.63161 .99811 L
F
.62904 .9933 m
.63161 .9933 L
.63161 .99811 L
.62904 .99811 L
F
.62391 .9933 m
.62648 .9933 L
.62648 .99811 L
.62391 .99811 L
F
.62648 .86343 m
.62904 .86343 L
.62904 .86824 L
.62648 .86824 L
F
.62904 .86343 m
.63161 .86343 L
.63161 .86824 L
.62904 .86824 L
F
.59922 .9933 m
.60084 .9933 L
.60084 .99811 L
.59922 .99811 L
F
.62391 .91153 m
.62648 .91153 L
.62648 .91634 L
.62391 .91634 L
F
.63161 .91153 m
.63417 .91153 L
.63417 .91634 L
.63161 .91634 L
F
.61366 1.02697 m
.61622 1.02697 L
.61622 1.03178 L
.61366 1.03178 L
F
.61109 1.02697 m
.61366 1.02697 L
.61366 1.03178 L
.61109 1.03178 L
F
.63673 .86343 m
.6393 .86343 L
.6393 .86824 L
.63673 .86824 L
F
.63417 .86343 m
.63673 .86343 L
.63673 .86824 L
.63417 .86824 L
F
.62648 .91153 m
.62904 .91153 L
.62904 .91634 L
.62648 .91634 L
F
.62904 .91153 m
.63161 .91153 L
.63161 .91634 L
.62904 .91634 L
F
.60084 .99811 m
.6034 .99811 L
.6034 1.00292 L
.60084 1.00292 L
F
.59922 .99811 m
.60084 .99811 L
.60084 1.00292 L
.59922 1.00292 L
F
.60853 .91153 m
.61109 .91153 L
.61109 .91634 L
.60853 .91634 L
F
.63417 1.01735 m
.63673 1.01735 L
.63673 1.02216 L
.63417 1.02216 L
F
.63161 1.01735 m
.63417 1.01735 L
.63417 1.02216 L
.63161 1.02216 L
F
.6034 .99811 m
.60597 .99811 L
.60597 1.00292 L
.6034 1.00292 L
F
.61366 .99811 m
.61622 .99811 L
.61622 1.00292 L
.61366 1.00292 L
F
.60853 .99811 m
.61109 .99811 L
.61109 1.00292 L
.60853 1.00292 L
F
.63673 .92596 m
.6393 .92596 L
.6393 .93077 L
.63673 .93077 L
F
.61109 .99811 m
.61366 .99811 L
.61366 1.00292 L
.61109 1.00292 L
F
.62391 1.01735 m
.62648 1.01735 L
.62648 1.02216 L
.62391 1.02216 L
F
.61879 .91634 m
.62135 .91634 L
.62135 .92115 L
.61879 .92115 L
F
.62648 1.01735 m
.62904 1.01735 L
.62904 1.02216 L
.62648 1.02216 L
F
.61622 .91634 m
.61879 .91634 L
.61879 .92115 L
.61622 .92115 L
F
.63673 .849 m
.6393 .849 L
.6393 .85381 L
.63673 .85381 L
F
.62391 1.02697 m
.62648 1.02697 L
.62648 1.03178 L
.62391 1.03178 L
F
.63673 1.01735 m
.6393 1.01735 L
.6393 1.02216 L
.63673 1.02216 L
F
.62904 1.01735 m
.63161 1.01735 L
.63161 1.02216 L
.62904 1.02216 L
F
.61879 1.01735 m
.62135 1.01735 L
.62135 1.02216 L
.61879 1.02216 L
F
.62135 1.01735 m
.62391 1.01735 L
.62391 1.02216 L
.62135 1.02216 L
F
.63417 .92596 m
.63673 .92596 L
.63673 .93077 L
.63417 .93077 L
F
.62648 .92596 m
.62904 .92596 L
.62904 .93077 L
.62648 .93077 L
F
.63161 .92596 m
.63417 .92596 L
.63417 .93077 L
.63161 .93077 L
F
.60597 .86343 m
.60853 .86343 L
.60853 .86824 L
.60597 .86824 L
F
.62904 .92596 m
.63161 .92596 L
.63161 .93077 L
.62904 .93077 L
F
.61879 .92596 m
.62135 .92596 L
.62135 .93077 L
.61879 .93077 L
F
.62391 .92596 m
.62648 .92596 L
.62648 .93077 L
.62391 .93077 L
F
.62135 .92596 m
.62391 .92596 L
.62391 .93077 L
.62135 .93077 L
F
.61622 .92596 m
.61879 .92596 L
.61879 .93077 L
.61622 .93077 L
F
.61366 .92596 m
.61622 .92596 L
.61622 .93077 L
.61366 .93077 L
F
.61879 .86343 m
.62135 .86343 L
.62135 .86824 L
.61879 .86824 L
F
.61109 .86343 m
.61366 .86343 L
.61366 .86824 L
.61109 .86824 L
F
.61622 .86343 m
.61879 .86343 L
.61879 .86824 L
.61622 .86824 L
F
.60597 .99811 m
.60853 .99811 L
.60853 1.00292 L
.60597 1.00292 L
F
.61366 .86343 m
.61622 .86343 L
.61622 .86824 L
.61366 .86824 L
F
.60853 .86343 m
.61109 .86343 L
.61109 .86824 L
.60853 .86824 L
F
.6034 .86343 m
.60597 .86343 L
.60597 .86824 L
.6034 .86824 L
F
.60084 .86343 m
.6034 .86343 L
.6034 .86824 L
.60084 .86824 L
F
.59922 .86343 m
.60084 .86343 L
.60084 .86824 L
.59922 .86824 L
F
.6034 .92596 m
.60597 .92596 L
.60597 .93077 L
.6034 .93077 L
F
.60597 .92596 m
.60853 .92596 L
.60853 .93077 L
.60597 .93077 L
F
.61109 .92596 m
.61366 .92596 L
.61366 .93077 L
.61109 .93077 L
F
.60084 1.00292 m
.6034 1.00292 L
.6034 1.00773 L
.60084 1.00773 L
F
.59922 1.00292 m
.60084 1.00292 L
.60084 1.00773 L
.59922 1.00773 L
F
.60084 .92596 m
.6034 .92596 L
.6034 .93077 L
.60084 .93077 L
F
.6034 .91153 m
.60597 .91153 L
.60597 .91634 L
.6034 .91634 L
F
.60597 .91153 m
.60853 .91153 L
.60853 .91634 L
.60597 .91634 L
F
.60853 .92596 m
.61109 .92596 L
.61109 .93077 L
.60853 .93077 L
F
.60853 1.02697 m
.61109 1.02697 L
.61109 1.03178 L
.60853 1.03178 L
F
.60597 1.02697 m
.60853 1.02697 L
.60853 1.03178 L
.60597 1.03178 L
F
.60597 .91634 m
.60853 .91634 L
.60853 .92115 L
.60597 .92115 L
F
.63673 .85862 m
.6393 .85862 L
.6393 .86343 L
.63673 .86343 L
F
.59922 .85862 m
.60084 .85862 L
.60084 .86343 L
.59922 .86343 L
F
.60853 .91634 m
.61109 .91634 L
.61109 .92115 L
.60853 .92115 L
F
.60597 1.00292 m
.60853 1.00292 L
.60853 1.00773 L
.60597 1.00773 L
F
.6034 1.00292 m
.60597 1.00292 L
.60597 1.00773 L
.6034 1.00773 L
F
.63417 .85862 m
.63673 .85862 L
.63673 .86343 L
.63417 .86343 L
F
.60084 .91153 m
.6034 .91153 L
.6034 .91634 L
.60084 .91634 L
F
.59922 .91153 m
.60084 .91153 L
.60084 .91634 L
.59922 .91634 L
F
.60853 1.00292 m
.61109 1.00292 L
.61109 1.00773 L
.60853 1.00773 L
F
.61622 1.00292 m
.61879 1.00292 L
.61879 1.00773 L
.61622 1.00773 L
F
.61109 1.00292 m
.61366 1.00292 L
.61366 1.00773 L
.61109 1.00773 L
F
.62648 .85862 m
.62904 .85862 L
.62904 .86343 L
.62648 .86343 L
F
.61366 1.00292 m
.61622 1.00292 L
.61622 1.00773 L
.61366 1.00773 L
F
.62391 1.00292 m
.62648 1.00292 L
.62648 1.00773 L
.62391 1.00773 L
F
.61879 1.00292 m
.62135 1.00292 L
.62135 1.00773 L
.61879 1.00773 L
F
.62135 1.00292 m
.62391 1.00292 L
.62391 1.00773 L
.62135 1.00773 L
F
.62648 1.00292 m
.62904 1.00292 L
.62904 1.00773 L
.62648 1.00773 L
F
.62904 1.00292 m
.63161 1.00292 L
.63161 1.00773 L
.62904 1.00773 L
F
.63161 1.00292 m
.63417 1.00292 L
.63417 1.00773 L
.63161 1.00773 L
F
.63161 .85862 m
.63417 .85862 L
.63417 .86343 L
.63161 .86343 L
F
.63417 1.00292 m
.63673 1.00292 L
.63673 1.00773 L
.63417 1.00773 L
F
.59922 .92596 m
.60084 .92596 L
.60084 .93077 L
.59922 .93077 L
F
.63673 1.00292 m
.6393 1.00292 L
.6393 1.00773 L
.63673 1.00773 L
F
.62904 .85862 m
.63161 .85862 L
.63161 .86343 L
.62904 .86343 L
F
.62391 .85862 m
.62648 .85862 L
.62648 .86343 L
.62391 .86343 L
F
.62135 .85862 m
.62391 .85862 L
.62391 .86343 L
.62135 .86343 L
F
.61879 .85862 m
.62135 .85862 L
.62135 .86343 L
.61879 .86343 L
F
.61879 .84418 m
.62135 .84418 L
.62135 .849 L
.61879 .849 L
F
.61366 .84418 m
.61622 .84418 L
.61622 .849 L
.61366 .849 L
F
.61622 .84418 m
.61879 .84418 L
.61879 .849 L
.61622 .849 L
F
.6034 1.02697 m
.60597 1.02697 L
.60597 1.03178 L
.6034 1.03178 L
F
.60084 1.02697 m
.6034 1.02697 L
.6034 1.03178 L
.60084 1.03178 L
F
.61366 .85862 m
.61622 .85862 L
.61622 .86343 L
.61366 .86343 L
F
.61622 .85862 m
.61879 .85862 L
.61879 .86343 L
.61622 .86343 L
F
.59922 .84418 m
.60084 .84418 L
.60084 .849 L
.59922 .849 L
F
.61109 .85862 m
.61366 .85862 L
.61366 .86343 L
.61109 .86343 L
F
.60853 .85862 m
.61109 .85862 L
.61109 .86343 L
.60853 .86343 L
F
.63417 .9933 m
.63673 .9933 L
.63673 .99811 L
.63417 .99811 L
F
.61622 .91153 m
.61879 .91153 L
.61879 .91634 L
.61622 .91634 L
F
.62135 .91153 m
.62391 .91153 L
.62391 .91634 L
.62135 .91634 L
F
.62648 .9933 m
.62904 .9933 L
.62904 .99811 L
.62648 .99811 L
F
.61879 .91153 m
.62135 .91153 L
.62135 .91634 L
.61879 .91634 L
F
.6034 .84418 m
.60597 .84418 L
.60597 .849 L
.6034 .849 L
F
.61109 .84418 m
.61366 .84418 L
.61366 .849 L
.61109 .849 L
F
.60597 .84418 m
.60853 .84418 L
.60853 .849 L
.60597 .849 L
F
.63673 .9933 m
.6393 .9933 L
.6393 .99811 L
.63673 .99811 L
F
.60853 .84418 m
.61109 .84418 L
.61109 .849 L
.60853 .849 L
F
.59922 1.00773 m
.60084 1.00773 L
.60084 1.01254 L
.59922 1.01254 L
F
.60084 1.00773 m
.6034 1.00773 L
.6034 1.01254 L
.60084 1.01254 L
F
.6034 .85862 m
.60597 .85862 L
.60597 .86343 L
.6034 .86343 L
F
.60597 1.00773 m
.60853 1.00773 L
.60853 1.01254 L
.60597 1.01254 L
F
.61366 .90672 m
.61622 .90672 L
.61622 .91153 L
.61366 .91153 L
F
.59922 .90672 m
.60084 .90672 L
.60084 .91153 L
.59922 .91153 L
F
.60597 .85862 m
.60853 .85862 L
.60853 .86343 L
.60597 .86343 L
F
.60084 .90672 m
.6034 .90672 L
.6034 .91153 L
.60084 .91153 L
F
.60084 .85862 m
.6034 .85862 L
.6034 .86343 L
.60084 .86343 L
F
.59922 1.02697 m
.60084 1.02697 L
.60084 1.03178 L
.59922 1.03178 L
F
.60084 .91634 m
.6034 .91634 L
.6034 .92115 L
.60084 .92115 L
F
.6034 .91634 m
.60597 .91634 L
.60597 .92115 L
.6034 .92115 L
F
.60853 .90672 m
.61109 .90672 L
.61109 .91153 L
.60853 .91153 L
F
.60084 .84418 m
.6034 .84418 L
.6034 .849 L
.60084 .849 L
F
.59922 .91634 m
.60084 .91634 L
.60084 .92115 L
.59922 .92115 L
F
.60597 .90672 m
.60853 .90672 L
.60853 .91153 L
.60597 .91153 L
F
.6034 1.00773 m
.60597 1.00773 L
.60597 1.01254 L
.6034 1.01254 L
F
.61109 .90672 m
.61366 .90672 L
.61366 .91153 L
.61109 .91153 L
F
.6034 .90672 m
.60597 .90672 L
.60597 .91153 L
.6034 .91153 L
F
.63417 1.01254 m
.63673 1.01254 L
.63673 1.01735 L
.63417 1.01735 L
F
.63161 1.01254 m
.63417 1.01254 L
.63417 1.01735 L
.63161 1.01735 L
F
.63673 .84418 m
.6393 .84418 L
.6393 .849 L
.63673 .849 L
F
.61366 .92115 m
.61622 .92115 L
.61622 .92596 L
.61366 .92596 L
F
.61622 .92115 m
.61879 .92115 L
.61879 .92596 L
.61622 .92596 L
F
.63673 1.01254 m
.6393 1.01254 L
.6393 1.01735 L
.63673 1.01735 L
F
.60597 .85381 m
.60853 .85381 L
.60853 .85862 L
.60597 .85862 L
F
.61109 .85381 m
.61366 .85381 L
.61366 .85862 L
.61109 .85862 L
F
.62904 1.01254 m
.63161 1.01254 L
.63161 1.01735 L
.62904 1.01735 L
F
.60853 .85381 m
.61109 .85381 L
.61109 .85862 L
.60853 .85862 L
F
.59922 .85381 m
.60084 .85381 L
.60084 .85862 L
.59922 .85862 L
F
.61366 1.01254 m
.61622 1.01254 L
.61622 1.01735 L
.61366 1.01735 L
F
.62135 .92115 m
.62391 .92115 L
.62391 .92596 L
.62135 .92596 L
F
.61109 1.01254 m
.61366 1.01254 L
.61366 1.01735 L
.61109 1.01735 L
F
.61622 1.01254 m
.61879 1.01254 L
.61879 1.01735 L
.61622 1.01735 L
F
.60853 .92115 m
.61109 .92115 L
.61109 .92596 L
.60853 .92596 L
F
.61109 .92115 m
.61366 .92115 L
.61366 .92596 L
.61109 .92596 L
F
.61879 .92115 m
.62135 .92115 L
.62135 .92596 L
.61879 .92596 L
F
.6034 .85381 m
.60597 .85381 L
.60597 .85862 L
.6034 .85862 L
F
.60084 .85381 m
.6034 .85381 L
.6034 .85862 L
.60084 .85862 L
F
.60853 1.00773 m
.61109 1.00773 L
.61109 1.01254 L
.60853 1.01254 L
F
.61622 1.00773 m
.61879 1.00773 L
.61879 1.01254 L
.61622 1.01254 L
F
.61109 1.00773 m
.61366 1.00773 L
.61366 1.01254 L
.61109 1.01254 L
F
.62135 .90672 m
.62391 .90672 L
.62391 .91153 L
.62135 .91153 L
F
.61366 1.00773 m
.61622 1.00773 L
.61622 1.01254 L
.61366 1.01254 L
F
.62391 1.00773 m
.62648 1.00773 L
.62648 1.01254 L
.62391 1.01254 L
F
.61879 1.00773 m
.62135 1.00773 L
.62135 1.01254 L
.61879 1.01254 L
F
.62135 1.00773 m
.62391 1.00773 L
.62391 1.01254 L
.62135 1.01254 L
F
.62648 1.00773 m
.62904 1.00773 L
.62904 1.01254 L
.62648 1.01254 L
F
.62904 1.00773 m
.63161 1.00773 L
.63161 1.01254 L
.62904 1.01254 L
F
.63161 1.00773 m
.63417 1.00773 L
.63417 1.01254 L
.63161 1.01254 L
F
.61622 .90672 m
.61879 .90672 L
.61879 .91153 L
.61622 .91153 L
F
.63417 1.00773 m
.63673 1.00773 L
.63673 1.01254 L
.63417 1.01254 L
F
.61366 .85381 m
.61622 .85381 L
.61622 .85862 L
.61366 .85862 L
F
.63673 1.00773 m
.6393 1.00773 L
.6393 1.01254 L
.63673 1.01254 L
F
.61879 .90672 m
.62135 .90672 L
.62135 .91153 L
.61879 .91153 L
F
.62391 .90672 m
.62648 .90672 L
.62648 .91153 L
.62391 .91153 L
F
.62648 .90672 m
.62904 .90672 L
.62904 .91153 L
.62648 .91153 L
F
.62904 .90672 m
.63161 .90672 L
.63161 .91153 L
.62904 .91153 L
F
.63161 .95963 m
.63417 .95963 L
.63417 .96444 L
.63161 .96444 L
F
.63417 .88267 m
.63673 .88267 L
.63673 .88748 L
.63417 .88748 L
F
.63417 .95963 m
.63673 .95963 L
.63673 .96444 L
.63417 .96444 L
F
.63673 .88267 m
.6393 .88267 L
.6393 .88748 L
.63673 .88748 L
F
.63161 .88267 m
.63417 .88267 L
.63417 .88748 L
.63161 .88748 L
F
.60084 .8971 m
.6034 .8971 L
.6034 .90191 L
.60084 .90191 L
F
.62648 .95963 m
.62904 .95963 L
.62904 .96444 L
.62648 .96444 L
F
.60597 .8971 m
.60853 .8971 L
.60853 .90191 L
.60597 .90191 L
F
.62391 .95963 m
.62648 .95963 L
.62648 .96444 L
.62391 .96444 L
F
.62904 .95963 m
.63161 .95963 L
.63161 .96444 L
.62904 .96444 L
F
.61622 .8971 m
.61879 .8971 L
.61879 .90191 L
.61622 .90191 L
F
.61366 .94039 m
.61622 .94039 L
.61622 .9452 L
.61366 .9452 L
F
.61879 .8971 m
.62135 .8971 L
.62135 .90191 L
.61879 .90191 L
F
.61622 .94039 m
.61879 .94039 L
.61879 .9452 L
.61622 .9452 L
F
.61109 .94039 m
.61366 .94039 L
.61366 .9452 L
.61109 .9452 L
F
.62904 .88267 m
.63161 .88267 L
.63161 .88748 L
.62904 .88748 L
F
.61109 .8971 m
.61366 .8971 L
.61366 .90191 L
.61109 .90191 L
F
.62648 .88267 m
.62904 .88267 L
.62904 .88748 L
.62648 .88748 L
F
.60853 .8971 m
.61109 .8971 L
.61109 .90191 L
.60853 .90191 L
F
.61366 .8971 m
.61622 .8971 L
.61622 .90191 L
.61366 .90191 L
F
.62648 .84418 m
.62904 .84418 L
.62904 .849 L
.62648 .849 L
F
.62904 .84418 m
.63161 .84418 L
.63161 .849 L
.62904 .849 L
F
.60597 1.01254 m
.60853 1.01254 L
.60853 1.01735 L
.60597 1.01735 L
F
.63417 .84418 m
.63673 .84418 L
.63673 .849 L
.63417 .849 L
F
.62391 .84418 m
.62648 .84418 L
.62648 .849 L
.62391 .849 L
F
.61879 1.01254 m
.62135 1.01254 L
.62135 1.01735 L
.61879 1.01735 L
F
.6034 1.01254 m
.60597 1.01254 L
.60597 1.01735 L
.6034 1.01735 L
F
.62135 1.01254 m
.62391 1.01254 L
.62391 1.01735 L
.62135 1.01735 L
F
.60853 1.01254 m
.61109 1.01254 L
.61109 1.01735 L
.60853 1.01735 L
F
.60084 1.01254 m
.6034 1.01254 L
.6034 1.01735 L
.60084 1.01735 L
F
.60084 .94039 m
.6034 .94039 L
.6034 .9452 L
.60084 .9452 L
F
.59922 .94039 m
.60084 .94039 L
.60084 .9452 L
.59922 .9452 L
F
.61109 .91634 m
.61366 .91634 L
.61366 .92115 L
.61109 .92115 L
F
.6034 .8971 m
.60597 .8971 L
.60597 .90191 L
.6034 .90191 L
F
.63673 .95963 m
.6393 .95963 L
.6393 .96444 L
.63673 .96444 L
F
.62648 1.01254 m
.62904 1.01254 L
.62904 1.01735 L
.62648 1.01735 L
F
.63161 .84418 m
.63417 .84418 L
.63417 .849 L
.63161 .849 L
F
.62135 .84418 m
.62391 .84418 L
.62391 .849 L
.62135 .849 L
F
.62391 1.01254 m
.62648 1.01254 L
.62648 1.01735 L
.62391 1.01735 L
F
.62648 1.02216 m
.62904 1.02216 L
.62904 1.02697 L
.62648 1.02697 L
F
.62135 1.02216 m
.62391 1.02216 L
.62391 1.02697 L
.62135 1.02697 L
F
.62391 1.02216 m
.62648 1.02216 L
.62648 1.02697 L
.62391 1.02697 L
F
.62904 1.02216 m
.63161 1.02216 L
.63161 1.02697 L
.62904 1.02697 L
F
.63161 1.02216 m
.63417 1.02216 L
.63417 1.02697 L
.63161 1.02697 L
F
.59922 1.01735 m
.60084 1.01735 L
.60084 1.02216 L
.59922 1.02216 L
F
.62648 1.02697 m
.62904 1.02697 L
.62904 1.03178 L
.62648 1.03178 L
F
.61366 1.01735 m
.61622 1.01735 L
.61622 1.02216 L
.61366 1.02216 L
F
.60084 1.01735 m
.6034 1.01735 L
.6034 1.02216 L
.60084 1.02216 L
F
.6034 1.01735 m
.60597 1.01735 L
.60597 1.02216 L
.6034 1.02216 L
F
.62648 .91634 m
.62904 .91634 L
.62904 .92115 L
.62648 .92115 L
F
.63673 .91634 m
.6393 .91634 L
.6393 .92115 L
.63673 .92115 L
F
.63161 .91634 m
.63417 .91634 L
.63417 .92115 L
.63161 .92115 L
F
.63673 1.02216 m
.6393 1.02216 L
.6393 1.02697 L
.63673 1.02697 L
F
.63417 .91634 m
.63673 .91634 L
.63673 .92115 L
.63417 .92115 L
F
.61109 1.02216 m
.61366 1.02216 L
.61366 1.02697 L
.61109 1.02697 L
F
.61879 1.02216 m
.62135 1.02216 L
.62135 1.02697 L
.61879 1.02697 L
F
.61366 1.02216 m
.61622 1.02216 L
.61622 1.02697 L
.61366 1.02697 L
F
.62904 .91634 m
.63161 .91634 L
.63161 .92115 L
.62904 .92115 L
F
.61622 1.02216 m
.61879 1.02216 L
.61879 1.02697 L
.61622 1.02697 L
F
.63673 1.02697 m
.6393 1.02697 L
.6393 1.03178 L
.63673 1.03178 L
F
.63417 1.02697 m
.63673 1.02697 L
.63673 1.03178 L
.63417 1.03178 L
F
.61622 1.02697 m
.61879 1.02697 L
.61879 1.03178 L
.61622 1.03178 L
F
.62904 1.02697 m
.63161 1.02697 L
.63161 1.03178 L
.62904 1.03178 L
F
.63673 .90672 m
.6393 .90672 L
.6393 .91153 L
.63673 .91153 L
F
.63417 .849 m
.63673 .849 L
.63673 .85381 L
.63417 .85381 L
F
.61879 1.02697 m
.62135 1.02697 L
.62135 1.03178 L
.61879 1.03178 L
F
.63161 .849 m
.63417 .849 L
.63417 .85381 L
.63161 .85381 L
F
.61366 .91634 m
.61622 .91634 L
.61622 .92115 L
.61366 .92115 L
F
.62135 1.02697 m
.62391 1.02697 L
.62391 1.03178 L
.62135 1.03178 L
F
.60853 1.01735 m
.61109 1.01735 L
.61109 1.02216 L
.60853 1.02216 L
F
.61109 1.01735 m
.61366 1.01735 L
.61366 1.02216 L
.61109 1.02216 L
F
.60084 .92115 m
.6034 .92115 L
.6034 .92596 L
.60084 .92596 L
F
.61622 1.01735 m
.61879 1.01735 L
.61879 1.02216 L
.61622 1.02216 L
F
.60597 1.01735 m
.60853 1.01735 L
.60853 1.02216 L
.60597 1.02216 L
F
.6034 .92115 m
.60597 .92115 L
.60597 .92596 L
.6034 .92596 L
F
.63161 1.02697 m
.63417 1.02697 L
.63417 1.03178 L
.63161 1.03178 L
F
.59922 .92115 m
.60084 .92115 L
.60084 .92596 L
.59922 .92596 L
F
.60597 .92115 m
.60853 .92115 L
.60853 .92596 L
.60597 .92596 L
F
.62391 .92115 m
.62648 .92115 L
.62648 .92596 L
.62391 .92596 L
F
.63417 .90672 m
.63673 .90672 L
.63673 .91153 L
.63417 .91153 L
F
.63673 .85381 m
.6393 .85381 L
.6393 .85862 L
.63673 .85862 L
F
.63417 .92115 m
.63673 .92115 L
.63673 .92596 L
.63417 .92596 L
F
.63161 .90672 m
.63417 .90672 L
.63417 .91153 L
.63161 .91153 L
F
.60084 .849 m
.6034 .849 L
.6034 .85381 L
.60084 .85381 L
F
.62904 .92115 m
.63161 .92115 L
.63161 .92596 L
.62904 .92596 L
F
.59922 .849 m
.60084 .849 L
.60084 .85381 L
.59922 .85381 L
F
.63161 .92115 m
.63417 .92115 L
.63417 .92596 L
.63161 .92596 L
F
.62648 .92115 m
.62904 .92115 L
.62904 .92596 L
.62648 .92596 L
F
.63161 .85381 m
.63417 .85381 L
.63417 .85862 L
.63161 .85862 L
F
.63417 .85381 m
.63673 .85381 L
.63673 .85862 L
.63417 .85862 L
F
.61622 .85381 m
.61879 .85381 L
.61879 .85862 L
.61622 .85862 L
F
.62904 .85381 m
.63161 .85381 L
.63161 .85862 L
.62904 .85862 L
F
.62648 .85381 m
.62904 .85381 L
.62904 .85862 L
.62648 .85862 L
F
.62135 .85381 m
.62391 .85381 L
.62391 .85862 L
.62135 .85862 L
F
.61879 .85381 m
.62135 .85381 L
.62135 .85862 L
.61879 .85862 L
F
.63673 .92115 m
.6393 .92115 L
.6393 .92596 L
.63673 .92596 L
F
.59922 1.01254 m
.60084 1.01254 L
.60084 1.01735 L
.59922 1.01735 L
F
.62391 .85381 m
.62648 .85381 L
.62648 .85862 L
.62391 .85862 L
F
.60853 .849 m
.61109 .849 L
.61109 .85381 L
.60853 .85381 L
F
.61109 .849 m
.61366 .849 L
.61366 .85381 L
.61109 .85381 L
F
.61879 .849 m
.62135 .849 L
.62135 .85381 L
.61879 .85381 L
F
.62904 .849 m
.63161 .849 L
.63161 .85381 L
.62904 .85381 L
F
.62648 .849 m
.62904 .849 L
.62904 .85381 L
.62648 .85381 L
F
.62391 .91634 m
.62648 .91634 L
.62648 .92115 L
.62391 .92115 L
F
.62135 .91634 m
.62391 .91634 L
.62391 .92115 L
.62135 .92115 L
F
.63417 1.02216 m
.63673 1.02216 L
.63673 1.02697 L
.63417 1.02697 L
F
.61366 .849 m
.61622 .849 L
.61622 .85381 L
.61366 .85381 L
F
.61622 .849 m
.61879 .849 L
.61879 .85381 L
.61622 .85381 L
F
.60853 1.02216 m
.61109 1.02216 L
.61109 1.02697 L
.60853 1.02697 L
F
.60597 1.02216 m
.60853 1.02216 L
.60853 1.02697 L
.60597 1.02697 L
F
.59922 1.02216 m
.60084 1.02216 L
.60084 1.02697 L
.59922 1.02697 L
F
.60597 .849 m
.60853 .849 L
.60853 .85381 L
.60597 .85381 L
F
.6034 .849 m
.60597 .849 L
.60597 .85381 L
.6034 .85381 L
F
.62391 .849 m
.62648 .849 L
.62648 .85381 L
.62391 .85381 L
F
.62135 .849 m
.62391 .849 L
.62391 .85381 L
.62135 .85381 L
F
.6034 1.02216 m
.60597 1.02216 L
.60597 1.02697 L
.6034 1.02697 L
F
.60084 1.02216 m
.6034 1.02216 L
.6034 1.02697 L
.60084 1.02697 L
F
0 g
.59922 .84418 m
.59922 .849 L
.59922 .85381 L
.59922 .85862 L
.59922 .86343 L
.59922 .86824 L
.59922 .87305 L
.59922 .87786 L
.59922 .88267 L
.59922 .88748 L
.59922 .89229 L
.59922 .8971 L
.59922 .90191 L
.59922 .90672 L
.59922 .91153 L
.59922 .91634 L
.59922 .92115 L
.59922 .92596 L
.59922 .93077 L
.59922 .93558 L
.59922 .94039 L
.59922 .9452 L
.59922 .95001 L
.59922 .95482 L
.59922 .95963 L
.59922 .96444 L
.59922 .96925 L
.59922 .97406 L
.59922 .97887 L
.59922 .98368 L
.59922 .98849 L
.59922 .9933 L
.59922 .99811 L
.59922 1.00292 L
.59922 1.00773 L
.59922 1.01254 L
.59922 1.01735 L
.59922 1.02216 L
.59922 1.02697 L
.59922 1.03178 L
s
.54186 .84418 m
.5393 .84418 L
.5393 .849 L
.5393 .85381 L
.5393 .85862 L
.5393 .86343 L
.5393 .86824 L
.5393 .87305 L
.5393 .87786 L
.5393 .88267 L
.5393 .88748 L
.5393 .89229 L
.5393 .8971 L
.5393 .90191 L
.5393 .90672 L
.5393 .91153 L
.5393 .91634 L
.5393 .92115 L
.5393 .92596 L
.5393 .93077 L
.5393 .93558 L
.5393 .94039 L
.5393 .9452 L
.5393 .95001 L
.5393 .95482 L
.5393 .95963 L
.5393 .96444 L
.5393 .96925 L
.5393 .97406 L
.5393 .97887 L
.5393 .98368 L
.5393 .98849 L
.5393 .9933 L
.5393 .99811 L
.5393 1.00292 L
.5393 1.00773 L
.5393 1.01254 L
.5393 1.01735 L
.5393 1.02216 L
.5393 1.02697 L
.5393 1.03178 L
.54186 1.03178 L
.54443 1.03178 L
.54699 1.03178 L
.54955 1.03178 L
.55212 1.03178 L
.55468 1.03178 L
.55725 1.03178 L
.55981 1.03178 L
.56238 1.03178 L
Mistroke
.56494 1.03178 L
.5675 1.03178 L
.57007 1.03178 L
.57263 1.03178 L
.5752 1.03178 L
.57776 1.03178 L
.58032 1.03178 L
.58289 1.03178 L
.58545 1.03178 L
.58802 1.03178 L
.59058 1.03178 L
.59314 1.03178 L
.59571 1.03178 L
.59827 1.03178 L
.59922 1.03178 L
Mfstroke
.54186 .84418 m
.54443 .84418 L
.54699 .84418 L
.54955 .84418 L
.55212 .84418 L
.55468 .84418 L
.55725 .84418 L
.55981 .84418 L
.56238 .84418 L
.56494 .84418 L
.5675 .84418 L
.57007 .84418 L
.57263 .84418 L
.5752 .84418 L
.57776 .84418 L
.58032 .84418 L
.58289 .84418 L
.58545 .84418 L
.58802 .84418 L
.59058 .84418 L
.59314 .84418 L
.59571 .84418 L
.59827 .84418 L
.59922 .84418 L
s
1 g
.23034 .79318 m
.25034 .79318 L
.25034 .81318 L
.23034 .81318 L
.23034 .79318 L
F
.23034 1.06279 m
.25034 1.06279 L
.25034 1.08279 L
.23034 1.08279 L
.23034 1.06279 L
F
.38155 .53625 m
.40155 .53625 L
.40155 .55625 L
.38155 .55625 L
.38155 .53625 L
F
.38155 1.31972 m
.40155 1.31972 L
.40155 1.33972 L
.38155 1.33972 L
.38155 1.31972 L
F
.74966 .33142 m
.76966 .33142 L
.76966 .35142 L
.74966 .35142 L
.74966 .33142 L
F
.74966 1.52455 m
.76966 1.52455 L
.76966 1.54455 L
.74966 1.54455 L
.74966 1.52455 L
F
0 g
.003 w
.23034 .79318 m
.25034 .79318 L
.25034 .81318 L
.23034 .81318 L
.23034 .79318 L
s
.23034 1.06279 m
.25034 1.06279 L
.25034 1.08279 L
.23034 1.08279 L
.23034 1.06279 L
s
.38155 .53625 m
.40155 .53625 L
.40155 .55625 L
.38155 .55625 L
.38155 .53625 L
s
.38155 1.31972 m
.40155 1.31972 L
.40155 1.33972 L
.38155 1.33972 L
.38155 1.31972 L
s
.74966 .33142 m
.76966 .33142 L
.76966 .35142 L
.74966 .35142 L
.74966 .33142 L
s
.74966 1.52455 m
.76966 1.52455 L
.76966 1.54455 L
.74966 1.54455 L
.74966 1.52455 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[54]:=",
  ImageSize->{153.5, 287.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40002I00017b000`400?l00000o`00003oV@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@000000
09P0Ng]k0@00000009P0Ng]k0@0000000700Ng]k1P00000R07]kN`400000001_07]kN`030000003o
ool0oooo00@0oooo00<0000007]kN`1kNg/07`1kNg/100000000KP1kNg/00`000000oooo0?ooo`06
0?ooo`030000001kNg/0Ng]k01h0Ng]k0@00000006h0Ng]k00@000000?ooo`3oool0oooo10000000
0`3oool0000007]kN`0O07]kN`400000001]07]kN`030000003oool0oooo0080oooo00H000000?oo
o`3oool000000?ooo`00000P07]kN`400000001]07]kN`030000003oool0oooo0080oooo00@00000
0?ooo`3oool000000P3oool00`000000Ng]k07]kN`0M07]kN`400000001]07]kN`030000003oool0
oooo0080oooo100000020?ooo`030000001kNg/0Ng]k01d0Ng]k0@00000006d0Ng]k00<000000?oo
o`3oool01`3oool00`000000Ng]k07]kN`0N07]kN`400000001^07]kN`030000003oool0oooo00H0
oooo00<0000007]kN`1kNg/07P1kNg/100000000KP1kNg/00`000000oooo0?ooo`050?ooo`030000
001kNg/0Ng]k01l0Ng]k0@00000006l0Ng]k1`00000R07]kN`400000002H07]kN`400000002H07]k
N`400000002H07]kN`400000002H07]kN`400000002H07]kN`400000002H07]kN`400000002H07]k
N`400000002H07]kN`400000000i07]kN`/00000E01kNg/100000000=@1kNg/4000000/0oooo1000
001@07]kN`400000000c07]kN`8000004`3oool2000004h0Ng]k0@0000000340Ng]k0P00000G0?oo
o`800000C01kNg/100000000;`1kNg/2000001/0oooo0P00001:07]kN`400000000^07]kN`030000
003oool0oooo01d0oooo0P00001807]kN`400000000]07]kN`030000003oool0oooo0200oooo0P00
001607]kN`400000000[07]kN`800000903oool00`000000Ng]k07]kN`1407]kN`400000000Z07]k
N`030000003oool0oooo02D0oooo00<0000007]kN`1kNg/0@`1kNg/100000000:@1kNg/00`000000
oooo0?ooo`0W0?ooo`030000001kNg/0Ng]k0480Ng]k0@00000002P0Ng]k00<000000?ooo`3oool0
:@3oool00`000000Ng]k07]kN`1107]kN`400000000W07]kN`030000003oool0oooo02/0oooo00<0
000007]kN`1kNg/0@01kNg/1000000009P1kNg/00`000000oooo0?ooo`0/0?ooo`030000001kNg/0
Ng]k0400Ng]k0@00000002H0Ng]k00<000000?ooo`3oool0;@3oool00`000000Ng]k07]kN`0o07]k
N`400000000U07]kN`030000003oool0oooo02l0oooo00<0000007]kN`1kNg/0?P1kNg/100000000
901kNg/00`000000oooo0?ooo`0C0?ooo`@000006@3oool00`000000Ng]k07]kN`0n07]kN`400000
000S07]kN`030000003oool0oooo01@0oooo00@000000?ooo`3oool000006P3oool00`000000Ng]k
07]kN`0m07]kN`400000000R07]kN`030000003oool0oooo01D0oooo00@000000?ooo`3oool00000
6P3oool00`000000Ng]k07]kN`0m07]kN`400000000Q07]kN`030000003oool0oooo01H0oooo1000
000J0?ooo`030000001kNg/0Ng]k03d0Ng]k0@0000000200Ng]k00<000000?ooo`3oool0=@3oool0
0`000000Ng]k07]kN`0m07]kN`400000000P07]kN`030000003oool0oooo03H0oooo00<0000007]k
N`1kNg/0?01kNg/1000000007`1kNg/00`000000oooo0?ooo`0g0?ooo`030000001kNg/0Ng]k03`0
Ng]k0@00000001h0Ng]k00<000000?ooo`3oool0>03oool00`000000Ng]k07]kN`0l07]kN`400000
000N07]kN`030000003oool0oooo03T0oooo00<0000007]kN`1kNg/0>`1kNg/1000000007@1kNg/0
0`000000oooo0?ooo`0j0?ooo`030000001kNg/0Ng]k03/0Ng]k0@00000001`0Ng]k00<000000?oo
o`3oool0>`3oool00`000000Ng]k07]kN`0k07]kN`400000000K07]kN`030000003oool0oooo03`0
oooo00<0000007]kN`1kNg/0>`1kNg/1000000006P1kNg/00`000000oooo0?ooo`0m0?ooo`030000
001kNg/0Ng]k03/0Ng]k0@00000001X0Ng]k00<000000?ooo`3oool0?@3oool00`000000Ng]k07]k
N`0k07]kN`400000000I07]kN`030000003oool0oooo03h0oooo00<0000007]kN`1kNg/0>`1kNg/1
00000000601kNg/00`000000oooo0?ooo`0o0?ooo`030000001kNg/0Ng]k03/0Ng]k0@00000001L0
Ng]k00<000000?ooo`3oool0@03oool00`000000Ng]k07]kN`0k07]kN`400000000F07]kN`030000
003oool0oooo0440oooo00<0000007]kN`1kNg/0>`1kNg/1000000005P1kNg/00`000000oooo0?oo
o`120?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000001@0Ng]k0P0000150?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000001@0Ng]k00<000000?ooo`3oool0A03oool00`000000Ng]k07]kN`0j07]k
N`400000000C07]kN`030000003oool0oooo04D0oooo00<0000007]kN`1kNg/0>P1kNg/100000000
4`1kNg/00`000000oooo0?ooo`150?ooo`030000001kNg/0Ng]k03X0Ng]k0@0000000180Ng]k00<0
00000?ooo`3oool0AP3oool00`000000Ng]k07]kN`0j07]kN`400000000B07]kN`030000003oool0
oooo04H0oooo00<0000007]kN`1kNg/0>P1kNg/1000000004@1kNg/00`000000oooo0?ooo`170?oo
o`030000001kNg/0Ng]k03X0Ng]k0@0000000140Ng]k00<000000?ooo`3oool0A`3oool00`000000
Ng]k07]kN`0j07]kN`400000000@07]kN`030000003oool0oooo04P0oooo00<0000007]kN`1kNg/0
>P1kNg/1000000003`1kNg/00`000000oooo0?ooo`190?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000l0Ng]k00<000000?ooo`3oool0B@3oool00`000000Ng]k07]kN`0j07]kN`400000000>07]k
N`030000003oool0oooo04X0oooo00<0000007]kN`1kNg/0>P1kNg/1000000003P1kNg/00`000000
oooo0?ooo`1:0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000d0Ng]k00<000000?ooo`3oool0
B`3oool00`000000Ng]k07]kN`0j07]kN`400000000=07]kN`030000003oool0oooo04/0oooo00<0
000007]kN`1kNg/0>P1kNg/1000000003@1kNg/00`000000oooo0?ooo`1;0?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000d0Ng]k00<000000?ooo`3oool0B`3oool00`000000Ng]k07]kN`0j07]k
N`400000000=07]kN`030000003oool0oooo04/0oooo00<0000007]kN`1kNg/0>P1kNg/100000000
301kNg/00`000000oooo0?ooo`1<0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000`0Ng]k00<0
00000?ooo`3oool0C03oool00`000000Ng]k07]kN`0j07]kN`400000000;07]kN`030000003oool0
oooo01D0oooo1000000d0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000/0Ng]k00<000000?oo
o`3oool05@3oool010000000oooo0?ooo`00000d0?ooo`030000001kNg/0Ng]k03X0Ng]k0@000000
00/0Ng]k00<000000?ooo`3oool05@3oool010000000oooo0?ooo`00000d0?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000/0Ng]k00<000000?ooo`3oool05@3oool4000003@0oooo00<0000007]k
N`1kNg/0>P1kNg/1000000002`1kNg/00`000000oooo0?ooo`1=0?ooo`030000001kNg/0Ng]k03X0
Ng]k0@00000000X0Ng]k00<000000?ooo`3oool0CP3oool00`000000Ng]k07]kN`0j07]kN`400000
000:07]kN`030000003oool0oooo04h0oooo00<0000007]kN`1kNg/0>P1kNg/1000000002P1kNg/0
0`000000oooo0?ooo`1>0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000X0Ng]k00<000000?oo
o`3oool0A@3oool:000003`0Ng]k0@00000000X0Ng]k00<000000?ooo`3oool0A@3oool00`000000
oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000X0Ng]k00<000000?ooo`3oool0
A@3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000X0Ng]k00<0
00000?ooo`3oool0A@3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?oo
o`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000
oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0
AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<0
00000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?oo
o`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000
oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0
AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<0
00000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?oo
o`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000
oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0
AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<0
00000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?oo
o`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000
oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0
AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000T0Ng]k00<0
00000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000T0Ng]k00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`060?oo
o`030000001kNg/0Ng]k03X0Ng]k0@00000000X0Ng]k00<000000?ooo`3oool0A@3oool00`000000
oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000X0Ng]k00<000000?ooo`3oool0
A@3oool00`000000oooo0?ooo`060?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000X0Ng]k00<0
00000?ooo`3oool0A@3oool:000003`0Ng]k0@00000000X0Ng]k00<000000?ooo`3oool0CP3oool0
0`000000Ng]k07]kN`0j07]kN`400000000:07]kN`030000003oool0oooo04h0oooo00<0000007]k
N`1kNg/0>P1kNg/1000000002P1kNg/00`000000oooo0?ooo`1>0?ooo`030000001kNg/0Ng]k03X0
Ng]k0@00000000X0Ng]k00<000000?ooo`3oool0CP3oool00`000000Ng]k07]kN`0j07]kN`400000
000;07]kN`030000003oool0oooo01D0oooo1000000d0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000/0Ng]k00<000000?ooo`3oool05@3oool010000000oooo0?ooo`00000d0?ooo`030000001k
Ng/0Ng]k03X0Ng]k0@00000000/0Ng]k00<000000?ooo`3oool05@3oool010000000oooo0?ooo`00
000d0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000/0Ng]k00<000000?ooo`3oool05@3oool4
000003@0oooo00<0000007]kN`1kNg/0>P1kNg/1000000002`1kNg/00`000000oooo0?ooo`1=0?oo
o`030000001kNg/0Ng]k03X0Ng]k0@00000000`0Ng]k00<000000?ooo`3oool0C03oool00`000000
Ng]k07]kN`0j07]kN`400000000<07]kN`030000003oool0oooo04`0oooo00<0000007]kN`1kNg/0
>P1kNg/1000000003@1kNg/00`000000oooo0?ooo`1;0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000000d0Ng]k00<000000?ooo`3oool0B`3oool00`000000Ng]k07]kN`0j07]kN`400000000=07]k
N`030000003oool0oooo04/0oooo00<0000007]kN`1kNg/0>P1kNg/1000000003@1kNg/00`000000
oooo0?ooo`1;0?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000000h0Ng]k00<000000?ooo`3oool0
BP3oool00`000000Ng]k07]kN`0j07]kN`400000000>07]kN`030000003oool0oooo04X0oooo00<0
000007]kN`1kNg/0>P1kNg/1000000003P1kNg/00`000000oooo0?ooo`1:0?ooo`030000001kNg/0
Ng]k03X0Ng]k0@00000000l0Ng]k00<000000?ooo`3oool0B@3oool00`000000Ng]k07]kN`0j07]k
N`400000000@07]kN`030000003oool0oooo04P0oooo00<0000007]kN`1kNg/0>P1kNg/100000000
401kNg/00`000000oooo0?ooo`180?ooo`030000001kNg/0Ng]k03X0Ng]k0@0000000140Ng]k00<0
00000?ooo`3oool0A`3oool00`000000Ng]k07]kN`0j07]kN`400000000A07]kN`030000003oool0
oooo04L0oooo00<0000007]kN`1kNg/0>P1kNg/1000000004P1kNg/00`000000oooo0?ooo`160?oo
o`030000001kNg/0Ng]k03X0Ng]k0@00000001<0Ng]k00<000000?ooo`3oool0A@3oool00`000000
Ng]k07]kN`0j07]kN`400000000C07]kN`030000003oool0oooo04D0oooo00<0000007]kN`1kNg/0
>P1kNg/100000000501kNg/00`000000oooo0?ooo`140?ooo`030000001kNg/0Ng]k03X0Ng]k0@00
000001@0Ng]k0P0000150?ooo`030000001kNg/0Ng]k03X0Ng]k0@00000001H0Ng]k00<000000?oo
o`3oool0@P3oool00`000000Ng]k07]kN`0j07]kN`400000000F07]kN`030000003oool0oooo0440
oooo00<0000007]kN`1kNg/0>`1kNg/1000000005`1kNg/00`000000oooo0?ooo`100?ooo`030000
001kNg/0Ng]k03/0Ng]k0@00000001P0Ng]k00<000000?ooo`3oool0?`3oool00`000000Ng]k07]k
N`0k07]kN`400000000I07]kN`030000003oool0oooo03h0oooo00<0000007]kN`1kNg/0>`1kNg/1
000000006@1kNg/00`000000oooo0?ooo`0n0?ooo`030000001kNg/0Ng]k03/0Ng]k0@00000001X0
Ng]k00<000000?ooo`3oool0?@3oool00`000000Ng]k07]kN`0k07]kN`400000000K07]kN`030000
003oool0oooo03`0oooo00<0000007]kN`1kNg/0>`1kNg/100000000701kNg/00`000000oooo0?oo
o`0k0?ooo`030000001kNg/0Ng]k03/0Ng]k0@00000001d0Ng]k00<000000?ooo`3oool0>P3oool0
0`000000Ng]k07]kN`0k07]kN`400000000M07]kN`030000003oool0oooo03X0oooo00<0000007]k
N`1kNg/0>`1kNg/1000000007P1kNg/00`000000oooo0?ooo`0i0?ooo`030000001kNg/0Ng]k03/0
Ng]k0@00000001l0Ng]k00<000000?ooo`3oool0=`3oool00`000000Ng]k07]kN`0l07]kN`400000
000P07]kN`030000003oool0oooo03H0oooo00<0000007]kN`1kNg/0?01kNg/100000000801kNg/0
0`000000oooo0?ooo`0f0?ooo`030000001kNg/0Ng]k03`0Ng]k0@0000000240Ng]k00<000000?oo
o`3oool05P3oool4000001X0oooo00<0000007]kN`1kNg/0?@1kNg/1000000008P1kNg/00`000000
oooo0?ooo`0E0?ooo`040000003oool0oooo000001X0oooo00<0000007]kN`1kNg/0?@1kNg/10000
00008`1kNg/00`000000oooo0?ooo`0D0?ooo`040000003oool0oooo000001X0oooo00<0000007]k
N`1kNg/0?@1kNg/100000000901kNg/00`000000oooo0?ooo`0C0?ooo`@000006@3oool00`000000
Ng]k07]kN`0n07]kN`400000000U07]kN`030000003oool0oooo02l0oooo00<0000007]kN`1kNg/0
?P1kNg/1000000009P1kNg/00`000000oooo0?ooo`0]0?ooo`030000001kNg/0Ng]k03l0Ng]k0@00
000002H0Ng]k00<000000?ooo`3oool0;03oool00`000000Ng]k07]kN`1007]kN`400000000W07]k
N`030000003oool0oooo02/0oooo00<0000007]kN`1kNg/0@01kNg/100000000:01kNg/00`000000
oooo0?ooo`0Y0?ooo`030000001kNg/0Ng]k0440Ng]k0@00000002T0Ng]k00<000000?ooo`3oool0
9`3oool00`000000Ng]k07]kN`1207]kN`400000000Z07]kN`030000003oool0oooo02D0oooo00<0
000007]kN`1kNg/0@`1kNg/100000000:`1kNg/2000002D0oooo00<0000007]kN`1kNg/0@`1kNg/1
00000000;01kNg/00`000000oooo0?ooo`0R0?ooo`030000001kNg/0Ng]k04@0Ng]k0@00000002d0
Ng]k0P00000O0?ooo`<00000A`1kNg/100000000;`1kNg/2000001/0oooo0P00001:07]kN`400000
000a07]kN`8000005`3oool2000004`0Ng]k0@00000003<0Ng]k0P00000C0?ooo`800000CP1kNg/1
00000000=@1kNg/4000000/0oooo1000001@07]kN`400000000i07]kN`/00000E01kNg/100000000
V01kNg/100000000V01kNg/100000000V01kNg/100000000V01kNg/100000000V01kNg/100000000
V01kNg/100000000V01kNg/100000000V01kNg/100000000K`1kNg/700000280Ng]k0@00000006h0
Ng]k00<000000?ooo`3oool01@3oool00`000000Ng]k07]kN`0O07]kN`400000001^07]kN`030000
003oool0oooo00H0oooo00<0000007]kN`1kNg/07P1kNg/100000000K@1kNg/00`000000oooo0?oo
o`070?ooo`030000001kNg/0Ng]k01h0Ng]k0@00000006d0Ng]k00<000000?ooo`3oool00P3oool4
00000080oooo00<0000007]kN`1kNg/07@1kNg/100000000K@1kNg/00`000000oooo0?ooo`020?oo
o`040000003oool0oooo00000080oooo00<0000007]kN`1kNg/07@1kNg/100000000K@1kNg/00`00
0000oooo0?ooo`020?ooo`040000003oool0oooo00000080oooo00<0000007]kN`1kNg/07@1kNg/1
00000000KP1kNg/010000000oooo0?ooo`3oool4000000030?ooo`000000Ng]k01l0Ng]k0@000000
06h0Ng]k00<000000?ooo`3oool01P3oool00`000000Ng]k07]kN`0N07]kN`400000001_07]kN`03
0000003oool0oooo00@0oooo00<0000007]kN`1kNg/07`1kNg/100000000L01kNg/600000280Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k
0@00000009P0Ng]k0@00000009P0Ng]k0@00000009P0Ng]k0@0000000001\
\>"],
  ImageRangeCache->{{{0, 152.5}, {286.938, 0}} -> {-3.69382, -5.80089, \
0.0404332, 0.0404332}}]
}, Open  ]],

Cell[TextData[{
  "Depending on the magnitude of \[Lambda] in (",
  
  CounterBox["NumberedEquation", "DahlquistEquation"],
  "), if we choose the step size ",
  Cell[BoxData[
      \(TraditionalForm\`h\)]],
  " such that ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(|\)\(R(
          h\ \[Lambda])\)\(|\)\(\ \)\(\(<\)\(\ \)\(1\)\)\)\)]],
  ", then errors in successive steps will be damped and the method is said to \
be absolutely stable."
}], "Text"],

Cell[TextData[{
  "If ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(|\)\(R(
          h\ \[Lambda])\)\(|\)\(\ \)\(\(>\)\(\ \)\(1\)\)\)\)]],
  ", then step size selection will be restricted by stability and not by \
local accuracy."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Stiffness detection", "Subsection"],

Cell[TextData[{
  "The device for stiffness detection that is used with the option ",
  StyleBox["StiffnessTest", "MR"],
  " is described within ",
  ButtonBox["StiffnessTest",
    ButtonData:>{"StiffnessTest.nb", None},
    ButtonStyle->"Hyperlink"],
  "."
}], "Text"],

Cell["\<\
Recast in terms of explicit Runge-Kutta methods the condition for \
stiffness detection can be formulated as:\
\>", "Text"],

Cell[BoxData[
    \(TraditionalForm\`\(\[Lambda]\& \[Tilde] \)\  = \ \
\[LeftDoubleBracketingBar]k\_s - k\_\(s - 1\)\[RightDoubleBracketingBar]\/\
\[LeftDoubleBracketingBar]g\_s - \ g\_\(s - \
1\)\[RightDoubleBracketingBar]\)], "NumberedEquation",
  CellTags->"StiffnessDevice"],

Cell[TextData[{
  "with ",
  Cell[BoxData[
      \(TraditionalForm\`g\_i\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`k\_i\)]],
  " defined in (",
  
  CounterBox["NumberedEquation", "ERKMethod"],
  ")."
}], "Text"],

Cell[TextData[{
  "The difference ",
  Cell[BoxData[
      \(TraditionalForm\`g\_s\)]],
  " - ",
  Cell[BoxData[
      \(TraditionalForm\`g\_\(s - 1\)\)]],
  " can be shown to correspond to a number of applications of the power \
method applied to ",
  Cell[BoxData[
      \(TraditionalForm\`h\ J\)]],
  ". "
}], "Text"],

Cell["\<\
The difference is therefore a good approximation of the eigenvector \
corresponding to the leading eigenvalue.\
\>", "Text"],

Cell[TextData[{
  "The product ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(|\)\(h\ \(\[Lambda]\& \[Tilde] \)\)\(|\)\)\)]],
  " gives an estimate that can be compared to the stability boundary in order \
to detect stiffness."
}], "Text"],

Cell[TextData[{
  "An ",
  Cell[BoxData[
      \(TraditionalForm\`s\)]],
  "-stage explicit Runge-Kutta has a form suitable for (",
  
  CounterBox["NumberedEquation", "StiffnessDevice"],
  ") if we require that",
  " ",
  Cell[BoxData[
      \(TraditionalForm\`c\_\(s - 1\)\  = \ \(c\_s = \ 1\)\)]],
  "."
}], "Text"],

Cell[BoxData[
    FormBox[
      TagBox[GridBox[{
            {"0", "0", "0", "\[CenterEllipsis]", "0", "0"},
            {\(c\_2\), \(a\_\(2, 1\)\), "0", "\[CenterEllipsis]", "0", "0"},
            {"\[VerticalEllipsis]", "\[VerticalEllipsis]", 
              "\[VerticalEllipsis]", "\[DescendingEllipsis]", 
              "\[VerticalEllipsis]", "\[VerticalEllipsis]"},
            {"1", \(a\_\(s - 1, 1\)\), \(a\_\(s - 1, 2\)\), 
              "\[CenterEllipsis]", "0", "0"},
            {"1", \(a\_\(s, 1\)\), \(a\_\(s, 2\)\), 
              "\[CenterEllipsis]", \(a\_\(s, s - 1\)\), "0"},
            {"", \(b\_1\), \(b\_2\), 
              "\[CenterEllipsis]", \(b\_\(s - 1\)\), \(b\_s\)}
            },
          RowLines->{False, False, False, False, True},
          ColumnLines->{True, False}],
        DisplayForm], TraditionalForm]], "NumberedEquation",
  CellTags->"StiffButcherTable"],

Cell[TextData[{
  "The default embedded pairs used in ",
  StyleBox["ExplicitRungeKutta", "MR"],
  " all have the form ",
  "(",
  
  CounterBox["NumberedEquation", "StiffButcherTable"],
  ")",
  "."
}], "Text"],

Cell[TextData[{
  "An important point is that (",
  
  CounterBox["NumberedEquation", "StiffnessDevice"],
  ") is very cheap and convenient; it uses already-available information from \
the integration and requires no additional function evaluations."
}], "Text"],

Cell[TextData[{
  "Another advantage of (",
  
  CounterBox["NumberedEquation", "StiffButcherTable"],
  ") is that it is straightforward to make use of consistent FSAL methods (",
  
  CounterBox["NumberedEquation", "FSALConditions"],
  ")."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Subsection"],

Cell["Select a stiff system modeling a chemical reaction.", "MathCaption"],

Cell[BoxData[
    \(system\  = \ GetNDSolveProblem["\<Robertson\>"]; \)], "Input",
  CellLabel->"In[55]:="],

Cell["\<\
This applies a built-in explicit Runge-Kutta method to the stiff \
system.\
\>", "MathCaption"],

Cell["\<\
By default stiffness detection is enabled, since it only has a \
small impact on the running time.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[system, \ Method \[Rule] "\<ExplicitRungeKutta\>"]\)], "Input",
  CellLabel->"In[56]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 0.3`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 0.3`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_3[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 0.3`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[56]="]
}, Open  ]],

Cell[TextData[{
  "The coefficients of the Dormand-Prince 5(4) pair are of the form (",
  
  CounterBox["NumberedEquation", "StiffButcherTable"],
  "). However, using these coefficients gives the following message."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[system, \ 
      Method \[Rule] {"\<ExplicitRungeKutta\>", 
          DifferenceOrder \[Rule] 5, "\<Coefficients\>" \[Rule] 
            DOPRICoefficients}]; \)], "Input",
  CellLabel->"In[57]:="],

Cell[BoxData[
    \(NDSolve::"mtdp" \(\(:\)\(\ \)\) 
      "\!\(DOPRICoefficients\) does not have a correctly defined property \
\!\(\"LinearStabilityBoundary\"\) in \!\(\"ExplicitRungeKutta\"\)."\)], \
"Message",
  CellLabel->"From In[57]:="],

Cell[BoxData[
    \(NDSolve::"initf" \(\(:\)\(\ \)\) 
      "The initialization of the method \!\(NDSolve`ExplicitRungeKutta\) \
failed."\)], "Message",
  CellLabel->"From In[57]:="]
}, Open  ]],

Cell["\<\
The reason is that the stiffness detection device needs to know \
where the border of the linear stability boundary is.\
\>", "Text"],

Cell["\<\
Therefore, we need to determine the point at which the stability \
region crosses the negative real axis.\
\>", "Text"],

Cell["\<\
We can set up an equation in terms of the linear stability function \
and solve it exactly to find the point where the contour crosses the negative \
real axis.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Reduce[Abs[DOPRIsf]\  \[Equal] \ 1\  && \ z\  < \ 0, z]\)], "Input",
  CellLabel->"In[58]:="],

Cell[BoxData[
    \(z \[Equal] 
      Root[600 + 300\ #1 + 100\ #1\^2 + 25\ #1\^3 + 5\ #1\^4 + #1\^5 &, 
        1]\)], "Output",
  CellLabel->"Out[58]="]
}, Open  ]],

Cell["\<\
In general, there may be more than one point of intersection and it \
may be necessary to choose the appropriate solution.\
\>", "Text"],

Cell["\<\
The following definition sets the value of the linear stability \
boundary.\
\>", "MathCaption"],

Cell[BoxData[
    \(\(DOPRICoefficients[5]\)["\<LinearStabilityBoundary\>"[]] = 
      Root[600 + 300*#1 + 100*#1^2 + 25*#1^3 + 5*#1^4 + #1^5 &, 1, 
        0]; \)], "Input",
  CellLabel->"In[59]:="],

Cell["\<\
The coefficients can now be used with stiffness detection \
enabled.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[system, \ 
      Method \[Rule] {"\<ExplicitRungeKutta\>", 
          DifferenceOrder \[Rule] 5, "\<Coefficients\>" \[Rule] 
            DOPRICoefficients}]\)], "Input",
  CellLabel->"In[60]:="],

Cell[BoxData[
    \(NDSolve::"ndstf" \(\(:\)\(\ \)\) 
      "At \!\(T\) == \!\(0.009515157777285154`\), system appears to be stiff. \
Methods Automatic, BDF or StiffnessSwitching may be more appropriate."\)], \
"Message",
  CellLabel->"From In[60]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              
              TagBox[\(InterpolatingFunction[{{0.`, 0.009515157777285154`}}, 
                  "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              
              TagBox[\(InterpolatingFunction[{{0.`, 0.009515157777285154`}}, 
                  "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_3[T]\), "\[Rule]", 
            RowBox[{
              
              TagBox[\(InterpolatingFunction[{{0.`, 0.009515157777285154`}}, 
                  "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[60]="]
}, Open  ]],

Cell[TextData[{
  "The Fehlberg 4(5) method does not have the correct coefficient structure \
(",
  
  CounterBox["NumberedEquation", "StiffButcherTable"],
  ") required for stiffness detection, since ",
  Cell[BoxData[
      \(TraditionalForm\`c\_s\  = 1/2 \[NotEqual] \ 1\)]],
  "."
}], "Text"],

Cell[TextData[{
  "The default value ",
  StyleBox["StiffnessTest\[Rule]Automatic", "MR"],
  " checks to see if the method coefficients provide a stiffness detection \
capability and if they do then stiffness detection is enabled."
}], "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Step control revisited", "Section",
  CellTags->"s:7"],

Cell[TextData[{
  "There are some reasons to look at alternatives to the standard Integral \
step controller (",
  
  CounterBox["NumberedEquation", "IController"],
  ") when considering mildly stiff problems."
}], "Text"],

Cell["This system models a chemical reaction.", "MathCaption"],

Cell[BoxData[
    \(system\  = \ GetNDSolveProblem["\<Robertson\>"]; \)], "Input",
  CellLabel->"In[61]:="],

Cell["\<\
This defines an explicit Runge-Kutta method based on the \
Dormand-Prince coefficients that does not use stiffness detection.\
\>", \
"MathCaption"],

Cell[BoxData[
    \(IERK\  = \ {"\<ExplicitRungeKutta\>", "\<Coefficients\>" \[Rule] 
          DOPRICoefficients, "\<DifferenceOrder\>" \[Rule] 
          5, "\<StiffnessTest\>" \[Rule] False}; \)], "Input",
  CellLabel->"In[62]:="],

Cell[TextData[{
  "This solves the system and plots the step sizes that are taken using the \
utility function ",
  StyleBox["StepDataPlot", "MR"],
  "."
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(isol\  = \ 
      NDSolve[system, \ Method \[Rule] IERK]; \), "\[IndentingNewLine]", 
    \(StepDataPlot[isol]; \)}], "Input",
  CellLabel->"In[63]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 3.1875 -0.185978 421.9 [
[.18318 .22342 -12 -9 ]
[.18318 .22342 12 0 ]
[.34256 .22342 -9 -9 ]
[.34256 .22342 9 0 ]
[.50193 .22342 -12 -9 ]
[.50193 .22342 12 0 ]
[.66131 .22342 -9 -9 ]
[.66131 .22342 9 0 ]
[.82068 .22342 -12 -9 ]
[.82068 .22342 12 0 ]
[.98006 .22342 -9 -9 ]
[.98006 .22342 9 0 ]
[.01131 .06716 -36 -4.5 ]
[.01131 .06716 0 4.5 ]
[.01131 .15154 -36 -4.5 ]
[.01131 .15154 0 4.5 ]
[.01131 .3203 -36 -4.5 ]
[.01131 .3203 0 4.5 ]
[.01131 .40468 -36 -4.5 ]
[.01131 .40468 0 4.5 ]
[.01131 .48906 -36 -4.5 ]
[.01131 .48906 0 4.5 ]
[.01131 .57344 -36 -4.5 ]
[.01131 .57344 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.18318 .23592 m
.18318 .24217 L
s
[(0.05)] .18318 .22342 0 1 Mshowa
.34256 .23592 m
.34256 .24217 L
s
[(0.1)] .34256 .22342 0 1 Mshowa
.50193 .23592 m
.50193 .24217 L
s
[(0.15)] .50193 .22342 0 1 Mshowa
.66131 .23592 m
.66131 .24217 L
s
[(0.2)] .66131 .22342 0 1 Mshowa
.82068 .23592 m
.82068 .24217 L
s
[(0.25)] .82068 .22342 0 1 Mshowa
.98006 .23592 m
.98006 .24217 L
s
[(0.3)] .98006 .22342 0 1 Mshowa
.125 Mabswid
.05568 .23592 m
.05568 .23967 L
s
.08756 .23592 m
.08756 .23967 L
s
.11943 .23592 m
.11943 .23967 L
s
.15131 .23592 m
.15131 .23967 L
s
.21506 .23592 m
.21506 .23967 L
s
.24693 .23592 m
.24693 .23967 L
s
.27881 .23592 m
.27881 .23967 L
s
.31068 .23592 m
.31068 .23967 L
s
.37443 .23592 m
.37443 .23967 L
s
.40631 .23592 m
.40631 .23967 L
s
.43818 .23592 m
.43818 .23967 L
s
.47006 .23592 m
.47006 .23967 L
s
.53381 .23592 m
.53381 .23967 L
s
.56568 .23592 m
.56568 .23967 L
s
.59756 .23592 m
.59756 .23967 L
s
.62943 .23592 m
.62943 .23967 L
s
.69318 .23592 m
.69318 .23967 L
s
.72506 .23592 m
.72506 .23967 L
s
.75693 .23592 m
.75693 .23967 L
s
.78881 .23592 m
.78881 .23967 L
s
.85256 .23592 m
.85256 .23967 L
s
.88443 .23592 m
.88443 .23967 L
s
.91631 .23592 m
.91631 .23967 L
s
.94818 .23592 m
.94818 .23967 L
s
.25 Mabswid
0 .23592 m
1 .23592 L
s
.02381 .06716 m
.03006 .06716 L
s
[(0.0006)] .01131 .06716 1 0 Mshowa
.02381 .15154 m
.03006 .15154 L
s
[(0.0008)] .01131 .15154 1 0 Mshowa
.02381 .3203 m
.03006 .3203 L
s
[(0.0012)] .01131 .3203 1 0 Mshowa
.02381 .40468 m
.03006 .40468 L
s
[(0.0014)] .01131 .40468 1 0 Mshowa
.02381 .48906 m
.03006 .48906 L
s
[(0.0016)] .01131 .48906 1 0 Mshowa
.02381 .57344 m
.03006 .57344 L
s
[(0.0018)] .01131 .57344 1 0 Mshowa
.125 Mabswid
.02381 .08826 m
.02756 .08826 L
s
.02381 .10935 m
.02756 .10935 L
s
.02381 .13045 m
.02756 .13045 L
s
.02381 .17264 m
.02756 .17264 L
s
.02381 .19373 m
.02756 .19373 L
s
.02381 .21483 m
.02756 .21483 L
s
.02381 .25702 m
.02756 .25702 L
s
.02381 .27811 m
.02756 .27811 L
s
.02381 .29921 m
.02756 .29921 L
s
.02381 .3414 m
.02756 .3414 L
s
.02381 .36249 m
.02756 .36249 L
s
.02381 .38359 m
.02756 .38359 L
s
.02381 .42578 m
.02756 .42578 L
s
.02381 .44687 m
.02756 .44687 L
s
.02381 .46797 m
.02756 .46797 L
s
.02381 .51016 m
.02756 .51016 L
s
.02381 .53125 m
.02756 .53125 L
s
.02381 .55235 m
.02756 .55235 L
s
.02381 .04607 m
.02756 .04607 L
s
.02381 .02497 m
.02756 .02497 L
s
.02381 .00388 m
.02756 .00388 L
s
.02381 .59454 m
.02756 .59454 L
s
.02381 .61563 m
.02756 .61563 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.5 Mabswid
.02381 .07224 m
.02576 .01472 L
.02728 .01472 L
.02879 .0172 L
.03033 .04423 L
.03207 .08734 L
.03413 .15227 L
.03669 .25105 L
.03999 .40512 L
.04446 .60332 L
.05042 .43369 L
.0551 .43369 L
.05978 .48856 L
.06488 .50078 L
.07007 .45122 L
.07488 .40146 L
.07932 .40871 L
.08381 .467 L
.08874 .51128 L
.09401 .48217 L
.09906 .41516 L
.1036 .39055 L
.10796 .43589 L
.11266 .50441 L
.11787 .51175 L
.12314 .44386 L
.1279 .38612 L
.13222 .40302 L
.13667 .47882 L
.1417 .52875 L
.1471 .40017 L
.15153 .40017 L
.15595 .47592 L
.16095 .52967 L
.16636 .40228 L
.17081 .40228 L
.17525 .47596 L
.18025 .52728 L
.18564 .40321 L
.19009 .40321 L
.19454 .47538 L
.19954 .52574 L
.20492 .40434 L
.20938 .40434 L
.21384 .47498 L
.21883 .52413 L
.22419 .48416 L
.22926 .40535 L
.23372 .38137 L
.23801 .43829 L
Mistroke
.24273 .51785 L
.24805 .52069 L
.25338 .43789 L
.2581 .37431 L
.26233 .39905 L
.26675 .48865 L
.27185 .54288 L
.27735 .38886 L
.2817 .38886 L
.28604 .48048 L
.29108 .54805 L
.29662 .39353 L
.301 .39353 L
.30538 .4813 L
.31042 .5432 L
.31593 .39464 L
.32031 .39464 L
.3247 .48018 L
.32973 .54092 L
.33523 .3964 L
.33963 .3964 L
.34403 .47957 L
.34905 .53832 L
.35453 .39787 L
.35894 .39787 L
.36335 .47891 L
.36837 .53603 L
.37383 .39929 L
.37825 .39929 L
.38267 .47831 L
.38769 .53387 L
.39313 .40061 L
.39756 .40061 L
.40199 .47775 L
.407 .53186 L
.41243 .40185 L
.41687 .40185 L
.42131 .47722 L
.42632 .52998 L
.43173 .40303 L
.43618 .40303 L
.44063 .47673 L
.44564 .5282 L
.45103 .40414 L
.45549 .40414 L
.45995 .47626 L
.46495 .52653 L
.47034 .4052 L
.4748 .4052 L
.47927 .47582 L
Mistroke
.48427 .52496 L
.48964 .485 L
.49471 .40621 L
.49918 .38222 L
.50348 .43913 L
.5082 .51867 L
.51352 .5215 L
.51887 .43872 L
.52359 .37514 L
.52783 .39988 L
.53225 .48947 L
.53736 .54369 L
.54287 .38967 L
.54722 .38967 L
.55157 .4813 L
.55661 .54886 L
.56216 .39432 L
.56654 .39432 L
.57093 .48211 L
.57598 .54401 L
.58149 .39541 L
.58588 .39541 L
.59028 .48098 L
.59531 .54173 L
.60081 .39715 L
.60522 .39715 L
.60962 .48036 L
.61466 .53914 L
.62014 .39861 L
.62455 .39861 L
.62897 .47969 L
.634 .53684 L
.63946 .40001 L
.64389 .40001 L
.64831 .47909 L
.65334 .53468 L
.65878 .40131 L
.66322 .40131 L
.66766 .47852 L
.67268 .53267 L
.67811 .40254 L
.68255 .40254 L
.687 .47798 L
.69202 .53078 L
.69743 .4037 L
.70189 .4037 L
.70634 .47748 L
.71135 .529 L
.71676 .4048 L
.72122 .4048 L
Mistroke
.72568 .477 L
.73069 .52733 L
.73608 .40585 L
.74055 .40585 L
.74502 .47655 L
.75003 .52574 L
.75541 .48573 L
.76048 .40684 L
.76496 .38283 L
.76926 .4398 L
.77398 .51944 L
.77931 .52227 L
.78466 .43938 L
.78939 .37572 L
.79363 .4005 L
.79806 .49019 L
.80317 .54447 L
.80869 .39026 L
.81304 .39026 L
.8174 .482 L
.82244 .54964 L
.828 .39489 L
.83239 .39489 L
.83678 .4828 L
.84183 .54478 L
.84735 .39598 L
.85175 .39598 L
.85615 .48166 L
.86119 .54249 L
.86669 .3977 L
.8711 .3977 L
.87551 .48103 L
.88055 .53989 L
.88604 .39915 L
.89046 .39915 L
.89488 .48035 L
.89991 .53759 L
.90538 .40054 L
.90981 .40054 L
.91424 .47974 L
.91927 .53542 L
.92472 .40183 L
.92916 .40183 L
.9336 .47916 L
.93863 .53339 L
.94406 .40305 L
.94851 .40305 L
.95296 .47861 L
.95798 .5315 L
.9634 .4042 L
Mistroke
.96786 .4042 L
.97232 .32598 L
.97619 .32598 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[63]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`009`3oool00`000000oooo0?ooo`3f0?ooo`009`3oool200000?L0oooo000W0?ooo`030000
003oool0oooo0?H0oooo000W0?ooo`030000003oool0oooo0?H0oooo000W0?ooo`800000m`3oool0
02L0oooo0P00003g0?ooo`009`3oool200000?L0oooo000W0?ooo`800000m`3oool002L0oooo0P00
003g0?ooo`009`3oool200000?L0oooo000W0?ooo`800000m`3oool002L0oooo0P00003g0?ooo`00
9`3oool200000?L0oooo000W0?ooo`800000m`3oool002L0oooo0P00003g0?ooo`000P3oool20000
00@0oooo00<000000?ooo`3oool00`3oool2000000@0oooo0P0000040?ooo`800000103oool20000
00D0oooo0P00003g0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo
00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000
003oool0oooo000000@0oooo0P00003g0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`04
0000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo000000@0oooo0`00003f0?ooo`0000D0oooo0000003oool0oooo
000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool01000
0000oooo0?ooo`0000020?ooo`<000001@3oool00`000000oooo0000003f0?ooo`0000D0oooo0000
003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000
0P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo00D0oooo00<000000?ooo`00
0000mP3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?ooo`0000020?oo
o`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?oo
o`050?ooo`030000003oool000000?H0oooo00020?ooo`8000002P3oool2000000@0oooo0P000004
0?ooo`800000103oool3000000@0oooo00<000000?ooo`000000mP3oool002L0oooo0`00003f0?oo
o`009`3oool00`000000oooo0000003f0?ooo`009`3oool00`000000oooo0000003f0?ooo`009`3o
ool00`000000oooo0000003f0?ooo`009`3oool00`000000oooo0000003f0?ooo`009`3oool30000
0?H0oooo000W0?ooo`030000003oool000000?H0oooo000W0?ooo`030000003oool000000?H0oooo
000W0?ooo`030000003oool000000?H0oooo000W0?ooo`030000003oool000000?H0oooo000W0?oo
o`<00000mP3oool002L0oooo00<000000?ooo`000000mP3oool002L0oooo00<000000?ooo`000000
mP3oool002L0oooo00<000000?ooo`000000mP3oool00080oooo0P0000040?ooo`030000003oool0
oooo00<0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000050?ooo`030000003oool0
00000?H0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3o
ool00000103oool00`000000oooo0000003f0?ooo`0000D0oooo0000003oool0oooo000000080?oo
o`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo000000@0oooo0`00003f0?ooo`0000D0oooo0000003oool0
oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool0
10000000oooo0?ooo`0000030?ooo`8000001@3oool00`000000oooo0000003f0?ooo`0000D0oooo
0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo00<00000
0?ooo`000000mP3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000
oooo0?ooo`0000040?ooo`030000003oool000000?H0oooo00020?ooo`8000002P3oool2000000@0
oooo0P0000040?ooo`800000103oool2000000D0oooo00<000000?ooo`000000mP3oool002L0oooo
0`00003f0?ooo`009`3oool00`000000oooo0000003f0?ooo`009`3oool00`000000oooo0000003f
0?ooo`009`3oool00`000000oooo0000003f0?ooo`009`3oool00`000000oooo0000003f0?ooo`00
9`3oool300000?H0oooo000W0?ooo`030000003oool0000001X0oooo0P0000040?ooo`030000003o
ool0oooo00<0oooo0P0000030?ooo`<000005`3oool2000000@0oooo00<000000?ooo`3oool00P3o
ool4000001H0oooo0P0000040?ooo`030000003oool0oooo0080oooo100000020?ooo`<000005`3o
ool2000000@0oooo00<000000?ooo`3oool00P3oool4000001H0oooo0P0000040?ooo`030000003o
ool0oooo0080oooo100000020?ooo`<000005`3oool2000000@0oooo00<000000?ooo`3oool00`3o
ool2000000<0oooo000W0?ooo`030000003oool0000001T0oooo00@000000?ooo`3oool00000203o
ool010000000oooo0?ooo`0000050?ooo`030000003oool0oooo01<0oooo00@000000?ooo`3oool0
00002P3oool00`000000oooo0?ooo`0D0?ooo`040000003oool0oooo000000X0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`0C0?ooo`040000003oool0oooo000000P0oooo00<00000
0?ooo`3oool05P3oool010000000oooo0?ooo`0000080?ooo`030000003oool0oooo00H0oooo00<0
00000?ooo`3oool04`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080
oooo000W0?ooo`040000003oool0oooo000001P0oooo00@000000?ooo`3oool00000203oool01000
0000oooo0?ooo`0000050?ooo`030000003oool0oooo01<0oooo00@000000?ooo`3oool000002P3o
ool00`000000oooo0?ooo`0D0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`0C0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3o
ool05@3oool010000000oooo0?ooo`0000090?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool04`3oool010000000oooo0?ooo`00000;0?ooo`4000000@3oool10?ooo`009`3oool01000
0000oooo0?ooo`00000H0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool00000
0P3oool3000001H0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`0D0?ooo`04
0000003oool0oooo000000X0oooo00@000000?ooo`3oool0oooo0`00000F0?ooo`040000003oool0
oooo000000X0oooo00<000000?ooo`3oool0503oool010000000oooo0?ooo`00000:0?ooo`040000
003oool0oooo0?ooo`<000005P3oool010000000oooo0?ooo`0000090?ooo`8000000`3oool002L0
oooo00@000000?ooo`3oool00000603oool010000000oooo0?ooo`0000080?ooo`040000003oool0
oooo00000080oooo00<000000?ooo`3oool05P3oool010000000oooo0?ooo`00000:0?ooo`030000
003oool0oooo01@0oooo00@000000?ooo`3oool000002P3oool01@000000oooo0?ooo`3oool00000
01P0oooo00@000000?ooo`3oool000002`3oool00`000000oooo0?ooo`0C0?ooo`040000003oool0
oooo000000/0oooo00@000000?ooo`3oool00000603oool010000000oooo0?ooo`00000;0?ooo`40
00000@3oool10?ooo`009`3oool2000000030?ooo`000000oooo01L0oooo00@000000?ooo`3oool0
0000203oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo01H0oooo00@000000?oo
o`3oool00000203oool3000001H0oooo00@000000?ooo`3oool00000203oool3000000<0oooo00<0
00000?ooo`3oool05P3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000001D0
oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`030000003oool0
oooo01H0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`009`3o
ool010000000oooo0?ooo`00000I0?ooo`8000002P3oool2000000<0oooo1000000F0?ooo`800000
2`3oool00`000000oooo0?ooo`0E0?ooo`8000002`3oool010000000oooo0?ooo`3oool4000001H0
oooo0P00000:0?ooo`8000005`3oool2000000X0oooo0P0000030?ooo`@000005P3oool2000000X0
oooo0P0000030?ooo`009`3oool010000000oooo0?ooo`00003e0?ooo`009`3oool010000000oooo
0?ooo`00003e0?ooo`009`3oool010000000oooo0?ooo`00003e0?ooo`008@3ooooj000000D0oooo
000W0?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00L0oooo000W0?ooo`040000003oool0oooo00000?D0oooo
000W0?ooo`040000003oool0oooo00000?D0oooo000W0?ooo`040000003oool0oooo00000?D0oooo
000W0?ooo`80000000<0oooo0000003oool0m03oool002L0oooo00@000000?ooo`3oool00000m@3o
ool002L0oooo00@000000?ooo`3oool00000m@3oool002L0oooo00@000000?ooo`3oool00000m@3o
ool002L0oooo00@000000?ooo`3oool00000m@3oool002L0oooo0P0000000`3oool000000?ooo`3d
0?ooo`009`3oool010000000oooo0?ooo`00003e0?ooo`009`3oool010000000oooo0?ooo`00003e
0?ooo`009`3oool010000000oooo0?ooo`00003e0?ooo`009`3oool010000000oooo0?ooo`00003e
0?ooo`009`3oool010000000oooo0?ooo`00003e0?ooo`009`3oool2000000030?ooo`000000oooo
0?@0oooo000W0?ooo`040000003oool0oooo00000?D0oooo000W0?ooo`040000003oool0oooo0000
0?D0oooo00020?ooo`800000103oool00`000000oooo0?ooo`030?ooo`800000103oool2000000<0
oooo100000020?ooo`@00000103oool010000000oooo0?ooo`00003e0?ooo`0000D0oooo0000003o
ool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103o
ool01@000000oooo0?ooo`3oool0000000L0oooo00@000000?ooo`3oool00000m@3oool000050?oo
o`000000oooo0?ooo`000000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
000000@0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`040?ooo`80000000<0oooo
0000003oool0m03oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`030?ooo`040000003oool0oooo00000>T0oooo0P00000:0?ooo`0000D0oooo0000003oool0
oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool00000j@3o
ool00`000000oooo0?ooo`090?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003o
ool0oooo00000080oooo00@000000?ooo`3oool000000P3oool3000000<0oooo00@000000?ooo`3o
ool00000103oool01@000000oooo0?ooo`3oool000000>P0oooo00<000000?ooo`3oool02@3oool0
0080oooo0P00000:0?ooo`800000103oool2000000D0oooo00<000000?ooo`3oool00P3oool20000
00D0oooo00D000000?ooo`3oool0oooo0000003X0?ooo`030000003oool0oooo00T0oooo000W0?oo
o`8000000P3oool00`000000oooo0?ooo`3V0?ooo`030000003oool0oooo00T0oooo000W0?ooo`05
0000003oool0oooo0?ooo`000000j03oool00`000000oooo0?ooo`090?ooo`009`3oool01@000000
oooo0?ooo`3oool000000>L0oooo00<000000?ooo`3oool02P3oool002L0oooo00D000000?ooo`3o
ool0oooo0000003W0?ooo`030000003oool0oooo00X0oooo000W0?ooo`050000003oool0oooo0?oo
o`000000i`3oool00`000000oooo0?ooo`0:0?ooo`009`3oool200000080oooo00<000000?ooo`3o
ool0i@3oool00`000000oooo0?ooo`0:0?ooo`009`3oool01@000000oooo0?ooo`3oool000000>L0
oooo00<000000?ooo`3oool02P3oool002L0oooo00D000000?ooo`3oool0oooo0000003W0?ooo`03
0000003oool0oooo00X0oooo000W0?ooo`050000003oool0oooo0?ooo`000000=@3oool00`000000
oooo0?ooo`2_0?ooo`030000003oool0oooo00X0oooo000W0?ooo`050000003oool0oooo0?ooo`00
0000=@3oool00`000000oooo0?ooo`0o0?ooo`030000003oool0oooo0400oooo00<000000?ooo`3o
ool0:P3oool00`000000oooo0?ooo`0:0?ooo`009`3oool01@000000oooo0?ooo`3oool0000002l0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0i0?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool0>P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo02X0
oooo00<000000?ooo`3oool02P3oool002L0oooo0P0000020?ooo`030000003oool0oooo01<0oooo
00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo03T0oooo
00<000000?ooo`3oool00`3oool2000003/0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`0Z0?ooo`030000003oool0oooo00X0oooo000W0?ooo`050000003oool0oooo0?ooo`000000
3`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0
0`3oool2000000<0oooo0P00000e0?ooo`030000003oool0oooo00<0oooo0P0000030?ooo`800000
=P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo0P00000U0?ooo`030000
003oool0oooo00/0oooo000W0?ooo`050000003oool0oooo0?ooo`0000003`3oool00`000000oooo
0?ooo`030?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool00`3oool2000000<0oooo
0P0000030?ooo`8000000`3oool2000002/0oooo00<000000?ooo`3oool00`3oool2000000<0oooo
0P0000030?ooo`800000<@3oool00`000000oooo0?ooo`030?ooo`8000000`3oool2000000<0oooo
0P00000P0?ooo`030000003oool0oooo00/0oooo00020?ooo`800000103oool00`000000oooo0?oo
o`030?ooo`800000103oool2000000<0oooo100000040?ooo`030000003oool0oooo00<0oooo00D0
00000?ooo`3oool0oooo0000000>0?ooo`8000001@3oool2000001L0oooo0P0000050?ooo`800000
0`3oool2000000<0oooo0P0000030?ooo`8000000P3oool300000080oooo0P00000Q0?ooo`800000
1@3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`800000
0`3oool200000240oooo0P0000050?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`800000
0`3oool2000001L0oooo00<000000?ooo`3oool02`3oool000050?ooo`000000oooo0?ooo`000000
203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`030?ooo`050000003oool0oooo0?ooo`0000001`3oool0
0`000000oooo0?ooo`040?ooo`800000103oool00`000000oooo000000020?ooo`<000000P3oool2
000000h0oooo0P0000050?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000P3oool0
0`000000oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000004P3oool2
000000D0oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool2
000000<0oooo0P0000020?ooo`<000000P3oool2000000<0oooo0P00000C0?ooo`8000001@3oool2
000000<0oooo0P0000030?ooo`8000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool2
000000<0oooo0P0000080?ooo`030000003oool0oooo00/0oooo00001@3oool000000?ooo`3oool0
000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`030000
003oool0oooo00D00000103oool200000080oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`040?ooo`800000103oool00`000000oooo000000020?ooo`030000003oool000000080oooo
0P0000030?ooo`8000000`3oool2000000@0oooo0P0000050?ooo`8000000`3oool2000000<0oooo
0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo
0P0000030?ooo`8000000`3oool200000080oooo0P0000030?ooo`800000103oool2000000D0oooo
00<000000?ooo`0000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool2000000<0oooo
0P0000020?ooo`030000003oool000000080oooo0P0000030?ooo`8000000`3oool2000000<0oooo
0P0000030?ooo`800000103oool2000000D0oooo0P0000030?ooo`8000000`3oool200000080oooo
0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080oooo
0P00000=0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool00000103oool010000000oooo0?ooo`0000020?ooo`030000003oool0
oooo00<0oooo00D000000?ooo`3oool0oooo000000070?ooo`8000001@3oool2000000@0oooo00<0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo0P00
00040?ooo`800000103oool00`000000oooo000000030?ooo`8000000`3oool2000000<0oooo0P00
00020?ooo`030000003oool000000080oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P00
00030?ooo`8000000P3oool2000000<0oooo0P0000040?ooo`800000103oool010000000oooo0?oo
o`0000020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080oooo
00<000000?ooo`0000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool2000000<0oooo
0P0000040?ooo`8000001@3oool2000000<0oooo0P0000030?ooo`8000000P3oool2000000<0oooo
0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000P3oool00`000000oooo
0?ooo`0<0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool00000103oool01`000000oooo0?ooo`3oool000000?ooo`0000001@3o
ool01@000000oooo0?ooo`3oool0000000L0oooo0P0000040?ooo`030000003oool0000000@0oooo
00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo
0P0000040?ooo`030000003oool0000000<0oooo00<000000?ooo`0000000`3oool2000000<0oooo
0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo
0P0000030?ooo`8000000`3oool200000080oooo0P0000030?ooo`800000103oool00`000000oooo
000000030?ooo`040000003oool0oooo00000080oooo0P0000030?ooo`8000000`3oool2000000<0
oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0
oooo0P0000030?ooo`8000000`3oool2000000@0oooo00<000000?ooo`0000000`3oool00`000000
oooo000000030?ooo`8000000`3oool200000080oooo0P0000030?ooo`8000000`3oool2000000<0
oooo0P0000030?ooo`8000000`3oool200000080oooo00<000000?ooo`3oool0303oool000050?oo
o`000000oooo0?ooo`000000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
00000080oooo0`0000040?ooo`8000001@3oool01@000000oooo0?ooo`3oool0000000L0oooo0P00
00040?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`0000000P3oool00`000000oooo
000000020?ooo`8000000`3oool2000000<0oooo0P0000040?ooo`030000003oool0000000<0oooo
00<000000?ooo`0000000`3oool2000000<0oooo0P0000030?ooo`8000000P3oool00`000000oooo
000000020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080oooo
0P0000030?ooo`800000103oool00`000000oooo000000030?ooo`040000003oool0oooo00000080
oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000P3oool00`000000
oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3oool2000000@0
oooo00<000000?ooo`0000000`3oool00`000000oooo000000030?ooo`8000000`3oool200000080
oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080
oooo00<000000?ooo`3oool0303oool00080oooo0P00000:0?ooo`800000103oool2000000D0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`050000003oool0oooo0?ooo`00
00001`3oool2000000@0oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo0P0000030?ooo`8000000`3oool2000000@0oooo00<000000?oo
o`0000000`3oool00`000000oooo000000030?ooo`8000000`3oool2000000<0oooo0P0000020?oo
o`030000003oool000000080oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?oo
o`8000000P3oool2000000<0oooo0P0000040?ooo`030000003oool0000000<0oooo00@000000?oo
o`3oool000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P000002
0?ooo`030000003oool000000080oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P000003
0?ooo`800000103oool00`000000oooo000000030?ooo`030000003oool0000000<0oooo0P000003
0?ooo`8000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo0P000003
0?ooo`8000000`3oool200000080oooo00<000000?ooo`3oool0303oool002L0oooo0P0000020?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`0000000`3oool010000000oooo0?ooo`000003
0?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P000003
0?ooo`800000103oool00`000000oooo000000030?ooo`030000003oool0000000<0oooo0P000003
0?ooo`8000000`3oool200000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P000003
0?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo0P000004
0?ooo`030000003oool0000000<0oooo00@000000?ooo`3oool000000P3oool2000000<0oooo0P00
00030?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo0P00
00030?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`800000103oool00`000000oooo0000
00030?ooo`030000003oool0000000<0oooo0P0000030?ooo`8000000P3oool00`000000oooo0000
00020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080oooo00<0
00000?ooo`3oool0303oool002L0oooo00D000000?ooo`3oool0oooo000000060?ooo`040000003o
ool0oooo000000<0oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000020?ooo`03
0000003oool000000080oooo0P0000030?ooo`8000000`3oool2000000@0oooo00<000000?ooo`00
00000`3oool00`000000oooo000000030?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`03
0000003oool000000080oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`80
00000P3oool00`000000oooo000000020?ooo`800000103oool00`000000oooo000000030?ooo`04
0000003oool0oooo00000080oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?oo
o`8000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000030?oo
o`8000000`3oool2000000<0oooo00@000000?ooo`3oool000000`3oool00`000000oooo00000002
0?ooo`030000003oool0000000<0oooo0P0000020?ooo`030000003oool000000080oooo0P000003
0?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000P3oool00`000000oooo0?ooo`0<
0?ooo`009`3oool01P000000oooo0?ooo`3oool000000?ooo`8000000`3oool010000000oooo0?oo
o`0000030?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool00`00
0000oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000040?ooo`030000003oool00000
00<0oooo00<000000?ooo`0000000`3oool200000080oooo00<000000?ooo`0000000P3oool00`00
0000oooo000000020?ooo`030000003oool000000080oooo0P0000030?ooo`8000000`3oool20000
00<0oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`800000103oool00`00
0000oooo000000030?ooo`040000003oool0oooo00000080oooo0P0000030?ooo`8000000`3oool2
000000<0oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`8000000`3oool2
000000<0oooo0P0000030?ooo`8000000`3oool2000000<0oooo00@000000?ooo`3oool000000`3o
ool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3o
ool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3o
ool200000080oooo00<000000?ooo`3oool0303oool002L0oooo00H000000?ooo`3oool0oooo0000
003oool2000000<0oooo00@000000?ooo`3oool000000`3oool010000000oooo0?ooo`0000020?oo
o`040000003oool0oooo00000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P000003
0?ooo`800000103oool00`000000oooo000000030?ooo`030000003oool0000000<0oooo00L00000
0?ooo`000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo
000000020?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo
00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`800000103oool00`000000oooo
000000030?ooo`040000003oool0oooo00000080oooo00<000000?ooo`0000000P3oool2000000<0
oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000
oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`030000003oool000000080
oooo00<000000?ooo`0000000`3oool010000000oooo0?ooo`0000030?ooo`030000003oool00000
0080oooo00P000000?ooo`3oool000000?ooo`000000oooo00000080oooo00<000000?ooo`000000
0P3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`000000
0`3oool200000080oooo00<000000?ooo`3oool0303oool002L0oooo00H000000?ooo`3oool0oooo
0000003oool2000000<0oooo00@000000?ooo`3oool000000`3oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo00000080oooo00P000000?ooo`3oool000000?ooo`000000oooo0000
0080oooo0P0000030?ooo`030000003oool0000000<0oooo00<000000?ooo`0000000`3oool00`00
0000oooo000000030?ooo`050000003oool000000?ooo`0000000P3oool00`000000oooo00000002
0?ooo`030000003oool000000080oooo00D000000?ooo`000000oooo000000020?ooo`030000003o
ool000000080oooo00D000000?ooo`000000oooo000000020?ooo`050000003oool000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`800000103oool00`000000oooo000000030?ooo`04
0000003oool0oooo00000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`050000003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`050000003oool0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P00
00020?ooo`030000003oool000000080oooo00<000000?ooo`0000000`3oool010000000oooo0?oo
o`0000030?ooo`030000003oool000000080oooo00H000000?ooo`3oool000000?ooo`0000020?oo
o`050000003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000P3oool0
0`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool0
0`000000oooo0?ooo`0<0?ooo`009`3oool200000080oooo00@000000?ooo`00000000000`3oool0
10000000oooo0?ooo`0000030?ooo`040000003oool0oooo00000080oooo00L000000?ooo`3oool0
oooo0000003oool000000080oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool0
00000080oooo00<000000?ooo`0000000`3oool00`000000oooo000000030?ooo`030000003oool0
000000<0oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo00<0
00000?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<000000?ooo`000000
0P3oool01@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool01@000000
oooo0000003oool000000080oooo00<000000?ooo`0000000`3oool00`000000oooo000000030?oo
o`040000003oool0oooo00000080oooo00<000000?ooo`0000000P3oool00`000000oooo00000002
0?ooo`050000003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003o
ool000000080oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo
00D000000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo00@000000?ooo`3o
ool000000P3oool01`000000oooo0?ooo`3oool000000?ooo`0000000P3oool01P000000oooo0?oo
o`000000oooo00000080oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool00000
0080oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo00<00000
0?ooo`0000000P3oool00`000000oooo0000000>0?ooo`009`3oool02@000000oooo0?ooo`3oool0
00000?ooo`000000oooo000000020?ooo`040000003oool0oooo000000<0oooo00@000000?ooo`3o
ool000000P3oool01`000000oooo0?ooo`3oool000000?ooo`0000000P3oool01@000000oooo0000
003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`040000
003oool0oooo000000<0oooo00@000000?ooo`3oool000000P3oool01@000000oooo0000003oool0
00000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`050000003oool0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`050000003oool000000?ooo`000000
0P3oool00`000000oooo000000020?ooo`050000003oool000000?ooo`0000000P3oool00`000000
oooo000000020?ooo`040000003oool0oooo000000<0oooo00T000000?ooo`3oool0oooo0000003o
ool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`050000003oool000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00D000000?ooo`00
0000oooo000000020?ooo`030000003oool000000080oooo00D000000?ooo`000000oooo00000002
0?ooo`030000003oool000000080oooo00@000000?ooo`3oool000000P3oool01@000000oooo0?oo
o`3oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`050000
003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`050000003oool000000?oo
o`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?oo
o`0000003P3oool002L0oooo00T000000?ooo`3oool0oooo0000003oool000000?ooo`0000000P3o
ool01`000000oooo0?ooo`3oool000000?ooo`0000000`3oool010000000oooo0?ooo`0000030?oo
o`030000003oool000000080oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool0
00000080oooo00<000000?ooo`0000000P3oool010000000oooo0?ooo`0000030?ooo`040000003o
ool0oooo00000080oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool000000080
oooo00<000000?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<000000?oo
o`0000000P3oool01@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool0
1@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool010000000oooo0?oo
o`0000030?ooo`090000003oool0oooo0?ooo`000000oooo0000003oool000000080oooo00<00000
0?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3o
ool00`000000oooo000000020?ooo`050000003oool000000?ooo`0000000P3oool00`000000oooo
000000020?ooo`050000003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`04
0000003oool0oooo00000080oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0
00000080oooo00<000000?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<0
00000?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<000000?ooo`000000
0P3oool00`000000oooo000000020?ooo`030000003oool0000000h0oooo000W0?ooo`090000003o
ool0oooo0?ooo`000000oooo0000003oool000000080oooo00L000000?ooo`3oool0oooo0000003o
ool0000000<0oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000020?ooo`050000
003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080
oooo00@000000?ooo`3oool000000`3oool010000000oooo0?ooo`0000020?ooo`050000003oool0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00D0
00000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo00D000000?ooo`000000
oooo000000020?ooo`030000003oool000000080oooo00D000000?ooo`000000oooo000000020?oo
o`030000003oool000000080oooo00@000000?ooo`3oool000000`3oool02@000000oooo0?ooo`3o
ool000000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo00D000000?ooo`00
0000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool01@00
0000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool01@000000oooo0000003o
ool000000080oooo00<000000?ooo`0000000P3oool010000000oooo0?ooo`0000020?ooo`050000
003oool0oooo0?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080
oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo00D000000?oo
o`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool0
0`000000oooo0000000>0?ooo`009`3oool02@000000oooo0?ooo`3oool000000?ooo`000000oooo
000000020?ooo`070000003oool0oooo0?ooo`000000oooo000000030?ooo`040000003oool0oooo
000000<0oooo00<000000?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`040000003oool0oooo000000<0oooo
00@000000?ooo`3oool000000P3oool01@000000oooo0000003oool000000080oooo00<000000?oo
o`0000000P3oool00`000000oooo000000020?ooo`050000003oool000000?ooo`0000000P3oool0
0`000000oooo000000020?ooo`050000003oool000000?ooo`0000000P3oool00`000000oooo0000
00020?ooo`030000003oool000000080oooo00D000000?ooo`000000oooo000000020?ooo`050000
003oool0oooo0?ooo`0000000P3oool02@000000oooo0?ooo`3oool000000?ooo`000000oooo0000
00020?ooo`030000003oool000000080oooo00D000000?ooo`000000oooo000000020?ooo`030000
003oool000000080oooo00<000000?ooo`0000000P3oool01@000000oooo0000003oool000000080
oooo00<000000?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<000000?oo
o`0000000P3oool010000000oooo0?ooo`0000020?ooo`050000003oool0oooo0?ooo`0000000P3o
ool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3o
ool01@000000oooo0000003oool000000080oooo00D000000?ooo`000000oooo000000020?ooo`03
0000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo0000000>0?ooo`00
9`3oool02@000000oooo0?ooo`3oool000000?ooo`000000oooo000000020?ooo`070000003oool0
oooo0?ooo`000000oooo000000030?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`00
00000P3oool01@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool00`00
0000oooo000000020?ooo`040000003oool0oooo000000<0oooo00@000000?ooo`3oool000000P3o
ool01@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo
000000020?ooo`050000003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`05
0000003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool00000
0080oooo00D000000?ooo`000000oooo000000020?ooo`050000003oool0oooo0?ooo`0000000P3o
ool02@000000oooo0?ooo`3oool000000?ooo`000000oooo000000020?ooo`030000003oool00000
0080oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool000000080oooo00<00000
0?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3o
ool01@000000oooo0000003oool000000080oooo00<000000?ooo`0000000P3oool010000000oooo
0?ooo`0000020?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`0000000P3oool00`00
0000oooo000000020?ooo`030000003oool000000080oooo00D000000?ooo`000000oooo00000002
0?ooo`050000003oool000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003o
ool000000080oooo00<000000?ooo`0000003P3oool002L0oooo0P0000020?ooo`050000003oool0
00000?ooo`0000000P3oool01`000000oooo0?ooo`3oool000000?ooo`0000000`3oool010000000
oooo0?ooo`0000030?ooo`8000000`3oool200000080oooo00<000000?ooo`0000000P3oool00`00
0000oooo000000020?ooo`040000003oool0oooo00000080oooo00D000000?ooo`3oool0oooo0000
00020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo0000
00020?ooo`030000003oool000000080oooo0P0000020?ooo`080000003oool0oooo0000003oool0
00000?ooo`0000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool20000
00<0oooo0P0000020?ooo`040000003oool0oooo000000<0oooo00@000000?ooo`3oool000000`3o
ool200000080oooo00H000000?ooo`3oool000000?ooo`0000020?ooo`050000003oool000000?oo
o`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P0000020?oo
o`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo00<000000?ooo`0000000`3oool010000000oooo0?ooo`000003
0?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo00000002
0?ooo`050000003oool000000?ooo`0000000P3oool01@000000oooo0000003oool000000080oooo
00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool0000000h0oooo
000W0?ooo`090000003oool0oooo0?ooo`000000oooo0000003oool000000080oooo00L000000?oo
o`3oool0oooo0000003oool0000000@0oooo00<000000?ooo`0000000`3oool2000000<0oooo0P00
00020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool010000000oooo0?oo
o`0000020?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo000000020?ooo`03
0000003oool000000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000020?ooo`03
0000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`03
0000003oool000000080oooo0P0000030?ooo`8000000P3oool010000000oooo0?ooo`0000030?oo
o`040000003oool0oooo000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<00000
0?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P000003
0?ooo`8000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<00000
0?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool0000000<0oooo00@00000
0?ooo`3oool000000`3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P00
00030?ooo`8000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000003`3oool002L0oooo00L00000
0?ooo`3oool0oooo0000003oool000000080oooo00<000000?ooo`0000000`3oool00`000000oooo
000000040?ooo`030000003oool0000000<0oooo0P0000030?ooo`8000000P3oool00`000000oooo
000000020?ooo`030000003oool000000080oooo00@000000?ooo`3oool000000P3oool01@000000
oooo0?ooo`3oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`8000000`3oool200000080oooo00@000000?ooo`3oool000000`3oool010000000oooo0?ooo`00
00030?ooo`8000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool200000080oooo00<0
00000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<0
00000?ooo`0000000P3oool00`000000oooo000000030?ooo`040000003oool0oooo000000<0oooo
00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool200000080oooo
00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo
00<000000?ooo`0000000P3oool2000000l0oooo00020?ooo`800000103oool00`000000oooo0?oo
o`030?ooo`800000103oool2000000<0oooo100000030?ooo`8000001@3oool01`000000oooo0?oo
o`3oool000000?ooo`0000000P3oool2000000@0oooo00<000000?ooo`000000103oool2000000@0
oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`030000003oool000000080
oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000030?ooo`030000003oool00000
0080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool20000
0080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000003oool00000
0080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000020?ooo`040000003oool0oooo
000000<0oooo00<000000?ooo`000000103oool200000080oooo00<000000?ooo`0000000P3oool0
0`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool2
000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool0
0`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000`3oool0
10000000oooo0?ooo`0000030?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3o
ool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3o
ool00`000000oooo000000020?ooo`030000003oool000000080oooo0P00000?0?ooo`0000D0oooo
0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
0000103oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool00P3oool01`00
0000oooo0?ooo`3oool000000?ooo`0000000P3oool2000000@0oooo00<000000?ooo`000000103o
ool2000000@0oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`030000003o
ool000000080oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000030?ooo`030000
003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`800000
0`3oool200000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000
003oool000000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000020?ooo`040000
003oool0oooo000000<0oooo00<000000?ooo`000000103oool200000080oooo00<000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`00
0000103oool00`000000oooo000000030?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P00000?0?ooo`00
00D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool00000103oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool00P3o
ool200000080oooo00<000000?ooo`0000000P3oool2000000@0oooo00<000000?ooo`000000103o
ool2000000@0oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?ooo`030000003o
ool000000080oooo00@000000?ooo`3oool000000`3oool00`000000oooo000000030?ooo`030000
003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`800000
0`3oool200000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`030000
003oool000000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000020?ooo`040000
003oool0oooo000000<0oooo00<000000?ooo`000000103oool200000080oooo00<000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?ooo`00
0000103oool00`000000oooo000000030?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`00
00000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P00000?0?ooo`00
00D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool00000103oool010000000oooo0?ooo`3oool3000000D0oooo00L000000?ooo`3oool0oooo
0000003oool000000080oooo0P0000050?ooo`800000103oool2000000@0oooo0P0000030?ooo`80
00000`3oool2000000<0oooo0P0000020?ooo`040000003oool0oooo000000<0oooo00<000000?oo
o`0000000`3oool00`000000oooo000000020?ooo`030000003oool000000080oooo00<000000?oo
o`0000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080oooo00<000000?oo
o`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo00<000000?oo
o`0000000`3oool00`000000oooo000000040?ooo`8000000P3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`8000000`3oool2000000<0oooo0P0000030?ooo`8000000P3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo00<000000?ooo`000000103oool00`000000oooo000000030?oo
o`030000003oool000000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000020?oo
o`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo0P00000?0?ooo`0000D0oooo0000003oool0oooo000000080?oo
o`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool01@000000oooo0?oo
o`3oool0000000L0oooo00L000000?ooo`3oool0oooo0000003oool0000000<0oooo00<000000?oo
o`3oool00`3oool2000000@0oooo0P0000040?ooo`8000000`3oool2000000<0oooo0P0000030?oo
o`8000000P3oool00`000000oooo000000040?ooo`030000003oool0000000<0oooo00<000000?oo
o`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P0000030?oo
o`8000000`3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo00<000000?oo
o`0000000P3oool2000000<0oooo0P0000030?ooo`030000003oool0000000<0oooo00<000000?oo
o`000000103oool200000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000030?oo
o`8000000`3oool200000080oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?oo
o`030000003oool0000000@0oooo00<000000?ooo`0000000`3oool00`000000oooo000000020?oo
o`030000003oool000000080oooo0P0000030?ooo`8000000`3oool200000080oooo00<000000?oo
o`0000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P00000?0?oo
o`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@00000
0?ooo`3oool000000P3oool3000000<0oooo00<000000?ooo`3oool01@3oool01`000000oooo0?oo
o`3oool000000?ooo`0000000`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00<0
oooo0P0000040?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000000`3oool2000000@0
oooo00<000000?ooo`0000000`3oool00`000000oooo000000020?ooo`030000003oool000000080
oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080
oooo0P0000030?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`030000003oool0000000@0
oooo0P0000040?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`030000003oool000000080
oooo00<000000?ooo`0000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool200000080
oooo0P0000030?ooo`8000000`3oool00`000000oooo000000040?ooo`030000003oool0000000<0
oooo00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0
oooo0P0000020?ooo`030000003oool000000080oooo00<000000?ooo`0000000P3oool00`000000
oooo000000020?ooo`8000003`3oool00080oooo0P00000:0?ooo`800000103oool2000000D0oooo
00<000000?ooo`3oool00P3oool3000000@0oooo00L000000?ooo`3oool0oooo0000003oool00000
00T0oooo00<000000?ooo`3oool00`3oool2000000D0oooo00<000000?ooo`3oool00P3oool01000
0000oooo0?ooo`3oool2000000<0oooo0P0000030?ooo`800000103oool00`000000oooo00000004
0?ooo`8000000P3oool00`000000oooo000000020?ooo`030000003oool000000080oooo0P000003
0?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`800000103oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo0080oooo00L000000?ooo`3oool0oooo0000003oool0000000@0oooo
0P0000040?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`030000003oool000000080oooo
00<000000?ooo`0000000P3oool2000000<0oooo0P0000030?ooo`8000000`3oool2000000<0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`800000103oool2000000@0oooo
00<000000?ooo`0000000P3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo
0P0000020?ooo`8000000`3oool2000000<0oooo0P0000030?ooo`8000003`3oool002L0oooo0P00
00030?ooo`8000002@3oool00`000000oooo0?ooo`030?ooo`8000001@3oool00`000000oooo0?oo
o`020?ooo`040000003oool0oooo0?ooo`8000000`3oool2000000<0oooo0P0000040?ooo`030000
003oool0000000@0oooo0P0000020?ooo`030000003oool000000080oooo0P0000030?ooo`800000
0`3oool2000000<0oooo0P0000030?ooo`8000000`3oool00`000000oooo0?ooo`020?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool00P3oool010000000oooo0?ooo`3oool2000000D0
oooo0P0000040?ooo`8000000`3oool2000000<0oooo0P0000020?ooo`8000000`3oool2000000<0
oooo0P0000040?ooo`040000003oool0oooo0?ooo`8000000`3oool2000000<0oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00<0oooo0P0000040?oo
o`8000000`3oool00`000000oooo000000020?ooo`8000000`3oool2000000<0oooo0P0000020?oo
o`8000000`3oool2000000@0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0=0?oo
o`009`3oool00`000000oooo0?ooo`020?ooo`8000002@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool00P3oool010000000oooo0?ooo`3oool20000
00<0oooo0P0000030?ooo`800000103oool00`000000oooo000000040?ooo`8000000P3oool20000
00<0oooo0P0000040?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool0oooo0P000003
0?ooo`8000000`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<00000
0?ooo`3oool00P3oool010000000oooo0?ooo`3oool2000000D0oooo0P0000040?ooo`8000000`3o
ool2000000<0oooo0P0000030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`020?ooo`040000003oool0oooo0?ooo`8000000`3oool2000000<0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00<0oooo
0P0000040?ooo`8000000`3oool2000000<0oooo0P0000040?ooo`040000003oool0oooo0?ooo`80
00000`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`0=0?ooo`009`3oool00`000000oooo0?ooo`020?ooo`800000
5P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
0P3oool01@000000oooo0?ooo`3oool0000000D0oooo00<000000?ooo`000000103oool2000000<0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080
oooo00@000000?ooo`3oool0oooo0P0000030?ooo`8000000`3oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo00<0oooo0P0000040?ooo`8000000`3oool2000000<0oooo0P0000030?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?oo
o`040000003oool0oooo0?ooo`800000103oool01@000000oooo0?ooo`3oool0000000@0oooo00<0
00000?ooo`3oool00P3oool00`000000oooo0?ooo`030?ooo`800000103oool2000000@0oooo00<0
00000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`040000003oool0oooo0?ooo`800000
0`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`0=0?ooo`009`3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo01D0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo0080oooo00D000000?ooo`3oool0oooo000000060?ooo`800000103oool2000000<0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo
00@000000?ooo`3oool0oooo0P0000040?ooo`050000003oool0oooo0?ooo`000000103oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool0oooo0P000003
0?ooo`8000000`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00D00000
0?ooo`3oool0oooo000000040?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00`3o
ool2000000D0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo0080oooo00@000000?ooo`3oool0oooo0P0000030?ooo`030000003oool0oooo0080oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00d0oooo
000W0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool05@3oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool01@000000oooo0?oo
o`3oool0000000`0oooo0P0000030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
0P3oool01@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool00P3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool02P3oool010000000
oooo0?ooo`3oool2000000<0oooo0P0000030?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool00P3oool01@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`020?ooo`040000003oool0oooo0?ooo`8000000`3oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?oo
o`0=0?ooo`009`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01D0oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01@0oooo00D00000
0?ooo`3oool0oooo000000040?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00D000000?ooo`3oool0oooo
000000040?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0503oool010000000oooo
0?ooo`3oool2000000<0oooo0P0000030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool00P3oool01@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`020?ooo`050000003oool0oooo0?ooo`000000103oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool03@3oool002L0oooo0P0000030?oo
o`030000003oool0oooo03H0oooo00D000000?ooo`3oool0oooo000000040?ooo`030000003oool0
oooo0080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo0080oooo00D000000?ooo`3oool0oooo0000000P0?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
7@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`020?ooo`050000003oool0oooo0?ooo`000000103oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool03@3oool002L0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0f0?ooo`050000003oool0oooo0?ooo`00
0000103oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool00P3oool01@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`0W0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00D000000?ooo`3oool0oooo00000004
0?ooo`030000003oool0oooo01L0oooo000W0?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool0=P3oool01@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0a0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool0
1@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo0200oooo000W0?ooo`030000003oool0oooo0080oooo00<00000
0?ooo`3oool0>P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo03H0oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo03D0oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo02X0oooo000W0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool0>P3oool00`000000oooo0?ooo`100?oo
o`030000003oool0oooo03l0oooo00<000000?ooo`3oool0;`3oool002L0oooo0P0000030?ooo`03
0000003oool0oooo0?40oooo000W0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
l@3oool002L0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3a0?ooo`000P3oool2
000000@0oooo00<000000?ooo`3oool00`3oool2000000@0oooo0P0000030?ooo`@000000`3oool2
000000D0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3a0?ooo`0000D0oooo0000
003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000
103oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool00P3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo0?40oooo00001@3oool000000?ooo`3oool0000000P0
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`020?ooo`8000000`3oool00`000000oooo0?oo
o`3a0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo
00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`8000001@3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo0?40oooo00001@3oool000000?ooo`3oool0000000P0
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool0l@3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo00000080oooo0`0000030?ooo`040000003oool0oooo0000
00@0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3a0?ooo`000P3oool2000000X0
oooo0P0000040?ooo`8000001@3oool00`000000oooo0?ooo`020?ooo`8000001@3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo0?40oooo000W0?ooo`8000000`3oool00`000000oooo
0?ooo`3a0?ooo`009`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0?40oooo000W
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0l@3oool002L0oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`3a0?ooo`009`3oool00`000000oooo0?ooo`3f0?ooo`00
9`3oool00`000000oooo0?ooo`3f0?ooo`009`3oool200000?L0oooo003o0?ooob40oooo003o0?oo
ob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?oo
ob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0491349, 0.000333824, \
0.00125589, 9.48836}}]
}, Open  ]],

Cell["\<\
Solving a stiff or mildly stiff problem with the standard step size \
controller leads to large oscillations, sometimes leading to a number of \
undesirable step size rejections.\
\>", "Text"],

Cell["The study of this issue is known as step-control stability.", "Text"],

Cell["\<\
It can be studied by matching the linear stability regions for the \
high- and low-sorder methods in an embedded pair.\
\>", "Text"],

Cell["\<\
One approach to addressing the oscillation is to derive special \
methods, but this compromises the local accuracy.\
\>", "Text"],

Cell[CellGroupData[{

Cell["PI step control", "Subsection"],

Cell[TextData[{
  "An appealing alternative to Integral step control (",
  
  CounterBox["NumberedEquation", "IController"],
  ") is Proportional-Integral or PI step control."
}], "Text"],

Cell["\<\
In this case the step size is selected using the local error in two \
successive integration steps according to the formula:\
\>", "Text"],

Cell[BoxData[
    \(TraditionalForm\`h\_\(n + 1\) = \ 
      h\_n\ \(\((Tol\/\[LeftDoubleBracketingBar]err\_n\
\[RightDoubleBracketingBar])\)\^\(k\_1/\ \(p\& \[Tilde] \)\)\) \((\
\[LeftDoubleBracketingBar]err\_\(n - 1\)\[RightDoubleBracketingBar]\/\
\[LeftDoubleBracketingBar]err\_n\[RightDoubleBracketingBar])\)\^\(k\_2\ /\ \
\(p\& \[Tilde] \)\)\)], "NumberedEquation",
  CellTags->"PIController"],

Cell["\<\
This has the effect of damping and hence gives a smoother step size \
sequence.\
\>", "Text"],

Cell[TextData[{
  "Note that Integral step control (",
  
  CounterBox["NumberedEquation", "IController"],
  ") is a special case of (",
  
  CounterBox["NumberedEquation", "PIController"],
  ") and is used if a step is rejected:"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`k\_1 = 1, 
    k\_2 = \(\(0\)\(\ \)\(.\)\)\)], "DisplayFormula"],

Cell[TextData[{
  "The option ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          FormBox[\(StepSizeControlParameters \[Rule] {k\_1\),
            "TraditionalForm"], ",", \(k\_2\)}], TraditionalForm]]],
  StyleBox["}", "MR"],
  " can be used to specify the values of ",
  Cell[BoxData[
      \(TraditionalForm\`k\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`k\_2\)]],
  "."
}], "Text"],

Cell[TextData[{
  "The option ",
  StyleBox["StepSizeStartingError", "MR"],
  " can be used to specify the initial scaled error estimate in (",
  
  CounterBox["NumberedEquation", "PIController"],
  "), which is needed in the first integration step."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Subsection"],

Cell[CellGroupData[{

Cell["Stiff problem", "Subsubsection"],

Cell[TextData[{
  "This defines a method similar to ",
  StyleBox["IERK", "MR"],
  " that uses the option ",
  StyleBox["StepSizeControlParameters", "MR"],
  " to specify a PI controller."
}], "Text"],

Cell["\<\
Here we use generic control parameters suggested by \
Gustafsson:\
\>", "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {\(\(k\_1 = \ 3/10\)\(,\)\), 
            RowBox[{\(k\_2\), "=", " ", 
              RowBox[{\(2/5\), 
                FormBox["",
                  "TraditionalForm"]}]}]}
          }], TraditionalForm]], "DisplayFormula"],

Cell["This specifies the step control parameters.", "MathCaption"],

Cell[BoxData[
    \(PIERK\  = {"\<ExplicitRungeKutta\>", "\<Coefficients\>" \[Rule] 
          DOPRICoefficients, "\<DifferenceOrder\>" \[Rule] 
          5, "\<StiffnessTest\>" \[Rule] 
          False, "\<StepSizeControlParameters\>" \[Rule] {3/10, \ 
            2/5}}; \)], "Input",
  CellLabel->"In[65]:="],

Cell["\<\
Solving the system again, it can be observed that the step size \
sequence is now much smoother.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(pisol\  = \ 
      NDSolve[system, \ Method \[Rule] PIERK]; \), "\[IndentingNewLine]", 
    \(StepDataPlot[pisol]; \)}], "Input",
  CellLabel->"In[66]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 3.18624 -0.0908389 359.989 [
[.18312 .25665 -12 -9 ]
[.18312 .25665 12 0 ]
[.34243 .25665 -9 -9 ]
[.34243 .25665 9 0 ]
[.50175 .25665 -12 -9 ]
[.50175 .25665 12 0 ]
[.66106 .25665 -9 -9 ]
[.66106 .25665 9 0 ]
[.82037 .25665 -12 -9 ]
[.82037 .25665 12 0 ]
[.97968 .25665 -9 -9 ]
[.97968 .25665 9 0 ]
[.01131 .08916 -36 -4.5 ]
[.01131 .08916 0 4.5 ]
[.01131 .17915 -42 -4.5 ]
[.01131 .17915 0 4.5 ]
[.01131 .35915 -42 -4.5 ]
[.01131 .35915 0 4.5 ]
[.01131 .44915 -36 -4.5 ]
[.01131 .44915 0 4.5 ]
[.01131 .53914 -42 -4.5 ]
[.01131 .53914 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.18312 .26915 m
.18312 .2754 L
s
[(0.05)] .18312 .25665 0 1 Mshowa
.34243 .26915 m
.34243 .2754 L
s
[(0.1)] .34243 .25665 0 1 Mshowa
.50175 .26915 m
.50175 .2754 L
s
[(0.15)] .50175 .25665 0 1 Mshowa
.66106 .26915 m
.66106 .2754 L
s
[(0.2)] .66106 .25665 0 1 Mshowa
.82037 .26915 m
.82037 .2754 L
s
[(0.25)] .82037 .25665 0 1 Mshowa
.97968 .26915 m
.97968 .2754 L
s
[(0.3)] .97968 .25665 0 1 Mshowa
.125 Mabswid
.05567 .26915 m
.05567 .2729 L
s
.08753 .26915 m
.08753 .2729 L
s
.1194 .26915 m
.1194 .2729 L
s
.15126 .26915 m
.15126 .2729 L
s
.21498 .26915 m
.21498 .2729 L
s
.24685 .26915 m
.24685 .2729 L
s
.27871 .26915 m
.27871 .2729 L
s
.31057 .26915 m
.31057 .2729 L
s
.3743 .26915 m
.3743 .2729 L
s
.40616 .26915 m
.40616 .2729 L
s
.43802 .26915 m
.43802 .2729 L
s
.46988 .26915 m
.46988 .2729 L
s
.53361 .26915 m
.53361 .2729 L
s
.56547 .26915 m
.56547 .2729 L
s
.59733 .26915 m
.59733 .2729 L
s
.6292 .26915 m
.6292 .2729 L
s
.69292 .26915 m
.69292 .2729 L
s
.72478 .26915 m
.72478 .2729 L
s
.75665 .26915 m
.75665 .2729 L
s
.78851 .26915 m
.78851 .2729 L
s
.85223 .26915 m
.85223 .2729 L
s
.8841 .26915 m
.8841 .2729 L
s
.91596 .26915 m
.91596 .2729 L
s
.94782 .26915 m
.94782 .2729 L
s
.25 Mabswid
0 .26915 m
1 .26915 L
s
.02381 .08916 m
.03006 .08916 L
s
[(0.0005)] .01131 .08916 1 0 Mshowa
.02381 .17915 m
.03006 .17915 L
s
[(0.00075)] .01131 .17915 1 0 Mshowa
.02381 .35915 m
.03006 .35915 L
s
[(0.00125)] .01131 .35915 1 0 Mshowa
.02381 .44915 m
.03006 .44915 L
s
[(0.0015)] .01131 .44915 1 0 Mshowa
.02381 .53914 m
.03006 .53914 L
s
[(0.00175)] .01131 .53914 1 0 Mshowa
.125 Mabswid
.02381 .01716 m
.02756 .01716 L
s
.02381 .03516 m
.02756 .03516 L
s
.02381 .05316 m
.02756 .05316 L
s
.02381 .07116 m
.02756 .07116 L
s
.02381 .10716 m
.02756 .10716 L
s
.02381 .12515 m
.02756 .12515 L
s
.02381 .14315 m
.02756 .14315 L
s
.02381 .16115 m
.02756 .16115 L
s
.02381 .19715 m
.02756 .19715 L
s
.02381 .21515 m
.02756 .21515 L
s
.02381 .23315 m
.02756 .23315 L
s
.02381 .25115 m
.02756 .25115 L
s
.02381 .28715 m
.02756 .28715 L
s
.02381 .30515 m
.02756 .30515 L
s
.02381 .32315 m
.02756 .32315 L
s
.02381 .34115 m
.02756 .34115 L
s
.02381 .37715 m
.02756 .37715 L
s
.02381 .39515 m
.02756 .39515 L
s
.02381 .41315 m
.02756 .41315 L
s
.02381 .43115 m
.02756 .43115 L
s
.02381 .46714 m
.02756 .46714 L
s
.02381 .48514 m
.02756 .48514 L
s
.02381 .50314 m
.02756 .50314 L
s
.02381 .52114 m
.02756 .52114 L
s
.02381 .55714 m
.02756 .55714 L
s
.02381 .57514 m
.02756 .57514 L
s
.02381 .59314 m
.02756 .59314 L
s
.02381 .61114 m
.02756 .61114 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.5 Mabswid
.02381 .12949 m
.02576 .01472 L
.02669 .05656 L
.028 .03007 L
.02907 .05216 L
.03033 .05289 L
.03161 .07411 L
.03307 .09002 L
.03467 .11995 L
.03653 .15429 L
.0387 .20639 L
.04133 .27696 L
.04459 .38184 L
.04877 .51416 L
.05413 .60332 L
.06027 .5904 L
.0663 .52549 L
.07176 .46596 L
.07668 .43264 L
.08132 .42371 L
.08587 .42961 L
.09048 .44111 L
.09519 .45163 L
.09999 .45771 L
.10484 .45905 L
.10971 .45734 L
.11456 .45469 L
.11939 .45257 L
.1242 .45153 L
.129 .45146 L
.1338 .45192 L
.1386 .4525 L
.14341 .45292 L
.14823 .45311 L
.15304 .45311 L
.15785 .45302 L
.16267 .45292 L
.16748 .45286 L
.17229 .45284 L
.17711 .45287 L
.18192 .45291 L
.18673 .45295 L
.19154 .45298 L
.19636 .45299 L
.20117 .453 L
.20598 .45301 L
.2108 .45302 L
.21561 .45304 L
.22042 .45305 L
.22524 .45307 L
Mistroke
.23005 .45308 L
.23487 .4531 L
.23968 .45311 L
.2445 .45312 L
.24931 .45314 L
.25413 .45315 L
.25894 .45317 L
.26375 .45318 L
.26857 .45319 L
.27339 .45321 L
.2782 .45322 L
.28302 .45324 L
.28783 .45325 L
.29265 .45326 L
.29746 .45328 L
.30228 .45329 L
.3071 .4533 L
.31191 .45332 L
.31673 .45333 L
.32154 .45335 L
.32636 .45336 L
.33118 .45337 L
.33599 .45339 L
.34081 .4534 L
.34563 .45341 L
.35044 .45342 L
.35526 .45344 L
.36008 .45345 L
.3649 .45346 L
.36971 .45348 L
.37453 .45349 L
.37935 .4535 L
.38417 .45352 L
.38899 .45353 L
.3938 .45354 L
.39862 .45355 L
.40344 .45357 L
.40826 .45358 L
.41308 .45359 L
.4179 .45361 L
.42272 .45362 L
.42753 .45363 L
.43235 .45364 L
.43717 .45365 L
.44199 .45367 L
.44681 .45368 L
.45163 .45369 L
.45645 .4537 L
.46127 .45372 L
.46609 .45373 L
Mistroke
.47091 .45374 L
.47573 .45375 L
.48055 .45376 L
.48537 .45378 L
.49019 .45379 L
.49501 .4538 L
.49983 .45381 L
.50465 .45382 L
.50947 .45384 L
.51429 .45385 L
.51912 .45386 L
.52394 .45387 L
.52876 .45388 L
.53358 .45389 L
.5384 .45391 L
.54322 .45392 L
.54804 .45393 L
.55287 .45394 L
.55769 .45395 L
.56251 .45396 L
.56733 .45397 L
.57215 .45398 L
.57698 .454 L
.5818 .45401 L
.58662 .45402 L
.59144 .45403 L
.59626 .45404 L
.60109 .45405 L
.60591 .45406 L
.61073 .45407 L
.61556 .45408 L
.62038 .45409 L
.6252 .4541 L
.63003 .45412 L
.63485 .45413 L
.63967 .45414 L
.6445 .45415 L
.64932 .45416 L
.65414 .45417 L
.65897 .45418 L
.66379 .45419 L
.66862 .4542 L
.67344 .45421 L
.67826 .45422 L
.68309 .45423 L
.68791 .45424 L
.69274 .45425 L
.69756 .45426 L
.70239 .45427 L
.70721 .45428 L
Mistroke
.71204 .45429 L
.71686 .4543 L
.72169 .45431 L
.72651 .45432 L
.73134 .45433 L
.73616 .45434 L
.74099 .45435 L
.74581 .45436 L
.75064 .45437 L
.75546 .45438 L
.76029 .45439 L
.76511 .4544 L
.76994 .45441 L
.77477 .45442 L
.77959 .45443 L
.78442 .45444 L
.78924 .45445 L
.79407 .45446 L
.7989 .45446 L
.80372 .45447 L
.80855 .45448 L
.81338 .45449 L
.8182 .4545 L
.82303 .45451 L
.82786 .45452 L
.83268 .45453 L
.83751 .45454 L
.84234 .45455 L
.84717 .45455 L
.85199 .45456 L
.85682 .45457 L
.86165 .45458 L
.86647 .45459 L
.8713 .4546 L
.87613 .45461 L
.88096 .45462 L
.88579 .45462 L
.89061 .45463 L
.89544 .45464 L
.90027 .45465 L
.9051 .45466 L
.90993 .45467 L
.91475 .45468 L
.91958 .45468 L
.92441 .45469 L
.92924 .4547 L
.93407 .45471 L
.9389 .45472 L
.94372 .45472 L
.94855 .45473 L
Mistroke
.95338 .45474 L
.95821 .45475 L
.96304 .45476 L
.96787 .45476 L
.9727 .30374 L
.97619 .30374 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[66]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00;@3oool00`000000oooo0?ooo`3`0?ooo`00;@3o
ool00`000000oooo0?ooo`3`0?ooo`00;@3oool00`000000oooo0?ooo`3`0?ooo`00;@3oool00`00
0000oooo0?ooo`3`0?ooo`00;@3oool00`000000oooo0?ooo`3`0?ooo`00;@3oool00`000000oooo
0?ooo`3`0?ooo`00;@3oool00`000000oooo0?ooo`3`0?ooo`00;@3oool200000?40oooo000]0?oo
o`800000l@3oool002d0oooo0P00003a0?ooo`00;@3oool200000?40oooo000]0?ooo`800000l@3o
ool002d0oooo0P00003a0?ooo`00;@3oool200000?40oooo000]0?ooo`800000l@3oool002d0oooo
0P00003a0?ooo`00;@3oool200000?40oooo000]0?ooo`800000l@3oool002d0oooo0P00003a0?oo
o`00203oool2000000@0oooo00<000000?ooo`3oool00`3oool2000000@0oooo0P0000040?ooo`80
00000`3oool3000000D0oooo0P00003a0?ooo`001`3oool010000000oooo0?ooo`0000080?ooo`04
0000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00050?ooo`030000003oool0oooo0080oooo0P00003a0?ooo`001`3oool010000000oooo0?ooo`00
00080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000
oooo0?ooo`0000050?ooo`030000003oool0oooo0080oooo0`00003`0?ooo`001`3oool010000000
oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000020?ooo`<000001@3oool00`000000oooo0000003`0?ooo`001`3o
ool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3o
ool000000P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`000000l03oool000L0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool00`000000
oooo0?ooo`050?ooo`030000003oool000000?00oooo00080?ooo`8000002P3oool2000000@0oooo
0P0000040?ooo`8000000`3oool4000000@0oooo00<000000?ooo`000000l03oool002d0oooo00<0
00000?ooo`000000l03oool002d0oooo00<000000?ooo`000000l03oool002d0oooo00<000000?oo
o`000000l03oool002d0oooo00<000000?ooo`000000l03oool002d0oooo00<000000?ooo`000000
l03oool002d0oooo00<000000?ooo`000000l03oool002d0oooo00<000000?ooo`000000l03oool0
02d0oooo00<000000?ooo`000000l03oool002d0oooo00@000000?ooo`3oool00000k`3oool002d0
oooo00@000000?ooo`3oool00000k`3oool002d0oooo00@000000?ooo`3oool00000k`3oool002d0
oooo00@000000?ooo`3oool00000k`3oool002d0oooo00@000000?ooo`3oool00000k`3oool002d0
oooo00@000000?ooo`3oool00000k`3oool002d0oooo00@000000?ooo`3oool00000k`3oool00080
oooo0P0000040?ooo`030000003oool0oooo00<0oooo0P0000040?ooo`800000103oool2000000@0
oooo00<000000?ooo`3oool00P3oool3000000D0oooo00@000000?ooo`3oool00000k`3oool00005
0?ooo`000000oooo0?ooo`000000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0
oooo00000080oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo0080oooo00@000000?ooo`3oool00000k`3oool000050?ooo`000000oooo0?ooo`00
0000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@00000
0?ooo`3oool00000103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0080oooo0P00
00000`3oool000000?ooo`3^0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003o
ool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?oo
o`040000003oool0oooo0?ooo`<000001@3oool010000000oooo0?ooo`00003_0?ooo`0000D0oooo
0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000050?ooo`040000003oool0oooo000000L0oooo00@00000
0?ooo`3oool00000k`3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool01000
0000oooo0?ooo`0000020?ooo`030000003oool0oooo00D0oooo00@000000?ooo`3oool00000k`3o
ool00080oooo0P00000:0?ooo`800000103oool2000000@0oooo0P0000030?ooo`@000000P3oool4
000000@0oooo00@000000?ooo`3oool00000k`3oool002d0oooo00@000000?ooo`3oool00000k`3o
ool002d0oooo00@000000?ooo`3oool00000k`3oool002d0oooo00@000000?ooo`3oool00000k`3o
ool002d0oooo00@000000?ooo`3oool00000k`3oool002d0oooo00@000000?ooo`3oool00000k`3o
ool002d0oooo00@000000?ooo`3oool00000k`3oool002d0oooo00@000000?ooo`3oool00000k`3o
ool002d0oooo00@000000?ooo`3oool00000603oool2000000@0oooo00<000000?ooo`3oool00`3o
ool2000000<0oooo0`00000F0?ooo`800000103oool00`000000oooo0?ooo`020?ooo`@000005@3o
ool2000000@0oooo00<000000?ooo`3oool00P3oool400000080oooo0`00000F0?ooo`800000103o
ool00`000000oooo0?ooo`020?ooo`@000005@3oool2000000@0oooo00<000000?ooo`3oool00P3o
ool400000080oooo0`00000F0?ooo`800000103oool00`000000oooo0?ooo`030?ooo`8000000`3o
ool002d0oooo00@000000?ooo`3oool000005`3oool010000000oooo0?ooo`0000080?ooo`040000
003oool0oooo000000D0oooo00<000000?ooo`3oool04P3oool010000000oooo0?ooo`00000:0?oo
o`030000003oool0oooo01<0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo0180oooo00@000000?ooo`3oool00000203oool00`000000oooo0?oo
o`0E0?ooo`040000003oool0oooo000000P0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`0B0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool002d0
oooo00@000000?ooo`3oool000005`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0
oooo000000D0oooo00<000000?ooo`3oool04P3oool010000000oooo0?ooo`00000:0?ooo`030000
003oool0oooo01<0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo0180oooo00@000000?ooo`3oool000002@3oool00`000000oooo0?ooo`0D0?oo
o`040000003oool0oooo000000T0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0B
0?ooo`040000003oool0oooo000000/0oooo0@0000010?ooo`40oooo000]0?ooo`040000003oool0
oooo000001L0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`<0
00005@3oool010000000oooo0?ooo`00000:0?ooo`030000003oool0oooo01<0oooo00@000000?oo
o`3oool000002P3oool010000000oooo0?ooo`3oool3000001D0oooo00@000000?ooo`3oool00000
2P3oool00`000000oooo0?ooo`0C0?ooo`040000003oool0oooo000000X0oooo00@000000?ooo`3o
ool0oooo0`00000E0?ooo`040000003oool0oooo000000T0oooo0P0000030?ooo`00;@3oool01@00
0000oooo0?ooo`3oool0000001H0oooo00@000000?ooo`3oool00000203oool010000000oooo0?oo
o`0000020?ooo`030000003oool0oooo01D0oooo00@000000?ooo`3oool000002P3oool00`000000
oooo0?ooo`0C0?ooo`040000003oool0oooo000000X0oooo00D000000?ooo`3oool0oooo0000000G
0?ooo`040000003oool0oooo000000/0oooo00<000000?ooo`3oool04P3oool010000000oooo0?oo
o`00000;0?ooo`040000003oool0oooo000001L0oooo00@000000?ooo`3oool000002`3oool10000
0040oooo0@3oool002d0oooo00D000000?ooo`3oool0oooo0000000F0?ooo`040000003oool0oooo
000000P0oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`0E0?ooo`040000003o
ool0oooo000000P0oooo0`00000E0?ooo`040000003oool0oooo000000P0oooo0`0000030?ooo`03
0000003oool0oooo01D0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`00000D
0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool00`000000oooo
0?ooo`0E0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool002d0
oooo00D000000?ooo`3oool0oooo0000000G0?ooo`8000002P3oool2000000<0oooo1000000E0?oo
o`8000002`3oool00`000000oooo0?ooo`0D0?ooo`8000002`3oool010000000oooo0?ooo`3oool4
000001D0oooo0P00000:0?ooo`8000005P3oool2000000X0oooo0P0000030?ooo`@000005@3oool2
000000X0oooo0P0000030?ooo`00;@3oool01@000000oooo0?ooo`3oool000000>h0oooo000]0?oo
o`050000003oool0oooo0?ooo`000000kP3oool002d0oooo00D000000?ooo`3oool0oooo0000003^
0?ooo`009`3ooood000000D0oooo000]0?ooo`050000003oool0oooo0?ooo`0000000P3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01`3oool002d0
oooo00D000000?ooo`3oool0oooo0000000Q0?ooo`030000003oool0oooo02@0oooo00<000000?oo
o`3oool0903oool00`000000oooo0?ooo`0T0?ooo`030000003oool0oooo02@0oooo00<000000?oo
o`3oool0903oool00`000000oooo0?ooo`070?ooo`00;@3oool01@000000oooo0?ooo`3oool00000
0>h0oooo000]0?ooo`050000003oool0oooo0?ooo`000000kP3oool002d0oooo00D000000?ooo`3o
ool0oooo0000003^0?ooo`00;@3oool01@000000oooo0?ooo`3oool000000>h0oooo000]0?ooo`05
0000003oool0oooo0?ooo`000000kP3oool002d0oooo00D000000?ooo`3oool0oooo0000003^0?oo
o`00;@3oool01@000000oooo0?ooo`3oool000000>80oooo0P00000:0?ooo`00;@3oool01@000000
oooo0?ooo`3oool000000>80oooo00<000000?ooo`3oool02@3oool002d0oooo00D000000?ooo`3o
ool0oooo0000003R0?ooo`030000003oool0oooo00T0oooo000]0?ooo`050000003oool0oooo0?oo
o`000000hP3oool00`000000oooo0?ooo`090?ooo`00;@3oool01@000000oooo0?ooo`3oool00000
0>80oooo00<000000?ooo`3oool02@3oool002d0oooo00D000000?ooo`3oool0oooo0000003R0?oo
o`030000003oool0oooo00T0oooo000]0?ooo`050000003oool0oooo0?ooo`000000hP3oool00`00
0000oooo0?ooo`090?ooo`00;@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0=l0
oooo00<000000?ooo`3oool02@3oool002d0oooo00<000000?ooo`3oool00P3oool00`000000oooo
0?ooo`3O0?ooo`030000003oool0oooo00T0oooo000]0?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool0g`3oool00`000000oooo0?ooo`090?ooo`00;@3oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo0=l0oooo00<000000?ooo`3oool02@3oool00080oooo0P0000040?oo
o`030000003oool0oooo00<0oooo0P0000040?ooo`8000000`3oool400000080oooo100000020?oo
o`<000001@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0=l0oooo00<000000?oo
o`3oool02@3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo000000@0oooo00D000000?ooo`3oool0oooo000000080?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3O0?ooo`030000
003oool0oooo00T0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool01@3oool00`000000oooo0?ooo`020?ooo`8000000`3oool00`000000oooo0?ooo`3O0?oo
o`030000003oool0oooo00T0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo00<0oooo00@0
00000?ooo`3oool0oooo0`0000050?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
g`3oool00`000000oooo0?ooo`090?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000
003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`040?oo
o`040000003oool0oooo000000L0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3O
0?ooo`030000003oool0oooo00T0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`<000000`3oool010000000oooo
0?ooo`0000020?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00P3oool00`000000
oooo0?ooo`3O0?ooo`030000003oool0oooo00T0oooo00020?ooo`8000002P3oool2000000@0oooo
0P0000050?ooo`030000003oool0oooo0080oooo0P0000030?ooo`@00000103oool00`000000oooo
0?ooo`020?ooo`030000003oool0oooo0=l0oooo00<000000?ooo`3oool02@3oool002d0oooo00<0
00000?ooo`3oool00P3oool00`000000oooo0?ooo`3O0?ooo`030000003oool0oooo00T0oooo000]
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0gP3oool00`000000oooo0?ooo`0:
0?ooo`00;@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0=h0oooo00<000000?oo
o`3oool02P3oool002d0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3N0?ooo`03
0000003oool0oooo00X0oooo000]0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
gP3oool00`000000oooo0?ooo`0:0?ooo`00;@3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo0=h0oooo00<000000?ooo`3oool02P3oool002d0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`3N0?ooo`030000003oool0oooo00X0oooo000]0?ooo`030000003oool0oooo
0080oooo00<000000?ooo`3oool0gP3oool00`000000oooo0?ooo`0:0?ooo`00;@3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo0=h0oooo00<000000?ooo`3oool02P3oool002d0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3N0?ooo`030000003oool0oooo00X0oooo
000]0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0gP3oool00`000000oooo0?oo
o`0:0?ooo`00;@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00H0oooo00<00000
0?ooo`3oool0e@3oool00`000000oooo0?ooo`0:0?ooo`00;@3oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`000000eP3oool00`000000oooo0?ooo`0:0?oo
o`00;@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00@0oooo00@000000?ooo`3o
ool00000eP3oool00`000000oooo0?ooo`0:0?ooo`00;@3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo00@0oooo00D000000?ooo`3oool0oooo0000003E0?ooo`030000003oool0oooo
00X0oooo00080?ooo`800000103oool00`000000oooo0?ooo`030?ooo`800000103oool2000000<0
oooo100000020?ooo`<000001@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00@0
oooo00D000000?ooo`3oool0oooo0000003E0?ooo`030000003oool0oooo00X0oooo00070?ooo`04
0000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo00@0oooo00D000000?ooo`3oool0oooo0000003E0?ooo`030000
003oool0oooo00X0oooo00070?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool00P3oool2000000<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo0=80oooo00<000000?ooo`3oool02P3oool000L0oooo00@000000?ooo`3o
ool00000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo00@0
00000?ooo`3oool0oooo0`0000050?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00<0oooo/`00000L0?ooo`030000
003oool0oooo00X0oooo00070?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000040?ooo`050000003oool0oooo0?ooo`0000001`3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0103oool0
0`000000oooo0000000200000;<0oooo7@00000<0?ooo`001`3oool010000000oooo0?ooo`000008
0?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool3000000<0oooo
00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo
00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`3M0?ooo`00203oool2000000X0oooo0P00
00040?ooo`8000001@3oool010000000oooo0?ooo`3oool4000000@0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0>D0oooo000]0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3U0?ooo`00;@3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0i@3oool0
02d0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo
0>D0oooo000]0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00P3oool00`000000
oooo0?ooo`3U0?ooo`00;@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo
00<000000?ooo`3oool0i@3oool002d0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo0>D0oooo000]0?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`3U0?ooo`00;@3oool00`000000oooo0?ooo`030?oo
o`050000003oool0oooo0?ooo`000000j03oool002d0oooo00<000000?ooo`3oool00`3oool01@00
0000oooo0?ooo`3oool000000>P0oooo000]0?ooo`030000003oool0oooo00<0oooo00D000000?oo
o`3oool0oooo0000003X0?ooo`00;@3oool00`000000oooo0?ooo`030?ooo`050000003oool0oooo
0?ooo`000000j03oool002d0oooo00<000000?ooo`3oool00`3oool01@000000oooo0?ooo`3oool0
00000>P0oooo000]0?ooo`030000003oool0oooo00<0oooo00D000000?ooo`3oool0oooo0000003X
0?ooo`00;@3oool00`000000oooo0?ooo`030?ooo`050000003oool0oooo0?ooo`000000j03oool0
02d0oooo00<000000?ooo`3oool00`3oool01@000000oooo0?ooo`3oool000000>P0oooo00020?oo
o`800000103oool00`000000oooo0?ooo`030?ooo`800000103oool2000000<0oooo100000030?oo
o`030000003oool0oooo0080oooo0`0000050?ooo`030000003oool0oooo00<0oooo00D000000?oo
o`3oool0oooo0000003X0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0
oooo00000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`030?ooo`050000
003oool0oooo0?ooo`000000j03oool000050?ooo`000000oooo0?ooo`000000203oool010000000
oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo0080oooo0P0000040?ooo`050000003oool0
oooo0?ooo`000000j03oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool00`3oool010000000
oooo0?ooo`3oool3000000D0oooo00<000000?ooo`3oool00`3oool01@000000oooo0?ooo`3oool0
00000>P0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool000000P3o
ool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo00@0oooo00@000000?ooo`3oool0
00001`3oool00`000000oooo0?ooo`030?ooo`040000003oool0oooo00000>T0oooo00001@3oool0
00000?ooo`3oool0000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`<000000`3oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo00D0oooo
00<000000?ooo`3oool00`3oool010000000oooo0?ooo`00003Y0?ooo`000P3oool2000000X0oooo
0P0000040?ooo`8000001@3oool010000000oooo0?ooo`3oool400000080oooo100000040?ooo`03
0000003oool0oooo00<0oooo00@000000?ooo`3oool00000j@3oool002d0oooo00<000000?ooo`3o
ool00`3oool010000000oooo0?ooo`00003Y0?ooo`00;@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool000000>T0oooo000]0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`000000
j@3oool002d0oooo00<000000?ooo`3oool0103oool00`000000oooo0000003Y0?ooo`00;@3oool0
0`000000oooo0?ooo`040?ooo`030000003oool000000>T0oooo000]0?ooo`030000003oool0oooo
00@0oooo0P00003Z0?ooo`00;@3oool00`000000oooo0?ooo`040?ooo`800000jP3oool002d0oooo
00<000000?ooo`3oool0103oool200000>X0oooo000]0?ooo`030000003oool0oooo00@0oooo0P00
003Z0?ooo`00;@3oool00`000000oooo0?ooo`040?ooo`800000jP3oool002d0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`3Y0?ooo`00;@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo0>T0oooo000]0?ooo`030000003oool0oooo0?00oooo000]0?ooo`030000003o
ool0oooo0?00oooo000]0?ooo`030000003oool0oooo0?00oooo003o0?ooob40oooo003o0?ooob40
oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40
oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40
oooo003o0?ooob40oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0580129, 0.000102336, \
0.00128782, 1.13984}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Nonstiff problem", "Subsubsection"],

Cell[TextData[{
  "In general the I step controller (",
  
  CounterBox["NumberedEquation", "IController"],
  ") is able to take larger steps for a nonstiff problem than the PI step \
controller (",
  
  CounterBox["NumberedEquation", "PIController"],
  ") as the following example illustrates."
}], "Text"],

Cell["\<\
Select and solve a non-stiff system using the I step \
controller.\
\>", "MathCaption"],

Cell[BoxData[
    \(system\  = \ GetNDSolveProblem["\<BrusselatorODE\>"]; \)], "Input",
  CellLabel->"In[68]:="],

Cell[CellGroupData[{

Cell[BoxData[{
    \(isol\  = \ 
      NDSolve[system, \ Method \[Rule] IERK]; \), "\[IndentingNewLine]", 
    \(StepDataPlot[isol]; \)}], "Input",
  CellLabel->"In[69]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0479274 -0.0199807 2.71425 [
[.26345 .23894 -3 -9 ]
[.26345 .23894 3 0 ]
[.50308 .23894 -6 -9 ]
[.50308 .23894 6 0 ]
[.74272 .23894 -6 -9 ]
[.74272 .23894 6 0 ]
[.98236 .23894 -6 -9 ]
[.98236 .23894 6 0 ]
[.01131 .11573 -24 -4.5 ]
[.01131 .11573 0 4.5 ]
[.01131 .38716 -24 -4.5 ]
[.01131 .38716 0 4.5 ]
[.01131 .52287 -18 -4.5 ]
[.01131 .52287 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.26345 .25144 m
.26345 .25769 L
s
[(5)] .26345 .23894 0 1 Mshowa
.50308 .25144 m
.50308 .25769 L
s
[(10)] .50308 .23894 0 1 Mshowa
.74272 .25144 m
.74272 .25769 L
s
[(15)] .74272 .23894 0 1 Mshowa
.98236 .25144 m
.98236 .25769 L
s
[(20)] .98236 .23894 0 1 Mshowa
.125 Mabswid
.07174 .25144 m
.07174 .25519 L
s
.11966 .25144 m
.11966 .25519 L
s
.16759 .25144 m
.16759 .25519 L
s
.21552 .25144 m
.21552 .25519 L
s
.31137 .25144 m
.31137 .25519 L
s
.3593 .25144 m
.3593 .25519 L
s
.40723 .25144 m
.40723 .25519 L
s
.45516 .25144 m
.45516 .25519 L
s
.55101 .25144 m
.55101 .25519 L
s
.59894 .25144 m
.59894 .25519 L
s
.64687 .25144 m
.64687 .25519 L
s
.69479 .25144 m
.69479 .25519 L
s
.79065 .25144 m
.79065 .25519 L
s
.83858 .25144 m
.83858 .25519 L
s
.8865 .25144 m
.8865 .25519 L
s
.93443 .25144 m
.93443 .25519 L
s
.25 Mabswid
0 .25144 m
1 .25144 L
s
.02381 .11573 m
.03006 .11573 L
s
[(0.05)] .01131 .11573 1 0 Mshowa
.02381 .38716 m
.03006 .38716 L
s
[(0.15)] .01131 .38716 1 0 Mshowa
.02381 .52287 m
.03006 .52287 L
s
[(0.2)] .01131 .52287 1 0 Mshowa
.125 Mabswid
.02381 .14287 m
.02756 .14287 L
s
.02381 .17002 m
.02756 .17002 L
s
.02381 .19716 m
.02756 .19716 L
s
.02381 .2243 m
.02756 .2243 L
s
.02381 .27859 m
.02756 .27859 L
s
.02381 .30573 m
.02756 .30573 L
s
.02381 .33287 m
.02756 .33287 L
s
.02381 .36001 m
.02756 .36001 L
s
.02381 .4143 m
.02756 .4143 L
s
.02381 .44144 m
.02756 .44144 L
s
.02381 .46859 m
.02756 .46859 L
s
.02381 .49573 m
.02756 .49573 L
s
.02381 .08859 m
.02756 .08859 L
s
.02381 .06145 m
.02756 .06145 L
s
.02381 .0343 m
.02756 .0343 L
s
.02381 .00716 m
.02756 .00716 L
s
.02381 .55001 m
.02756 .55001 L
s
.02381 .57716 m
.02756 .57716 L
s
.02381 .6043 m
.02756 .6043 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.5 Mabswid
.02381 .10297 m
.02598 .15963 L
.02915 .123 L
.03168 .12075 L
.03416 .1077 L
.03642 .10082 L
.03855 .09755 L
.04062 .09298 L
.04262 .09081 L
.04458 .09156 L
.04655 .09607 L
.04859 .10663 L
.05083 .13268 L
.05353 .14444 L
.05643 .13777 L
.05921 .13975 L
.06203 .13536 L
.06478 .12908 L
.06741 .1284 L
.07003 .13003 L
.07268 .13328 L
.07538 .13778 L
.07817 .14335 L
.08105 .15 L
.08406 .15781 L
.0872 .16697 L
.0905 .17775 L
.09399 .19051 L
.0977 .20567 L
.10169 .22364 L
.10599 .24452 L
.11066 .26735 L
.11573 .28876 L
.12119 .29255 L
.12671 .26374 L
.13172 .24303 L
.13636 .23059 L
.14078 .22327 L
.14508 .2194 L
.14931 .21809 L
.15351 .21886 L
.15773 .22145 L
.16199 .22571 L
.16633 .23159 L
.17077 .23912 L
.17535 .24838 L
.18008 .25954 L
.18502 .27284 L
.19019 .28863 L
.19564 .3074 L
Mistroke
.20142 .32992 L
.2076 .35745 L
.21426 .39231 L
.22154 .43985 L
.22966 .51858 L
.23917 .58904 L
.24993 .56241 L
.26021 .50395 L
.26946 .48115 L
.27831 .47209 L
.287 .47788 L
.29579 .50887 L
.30513 .41666 L
.31284 .34492 L
.31928 .28141 L
.3246 .22759 L
.32898 .19127 L
.33271 .16353 L
.33595 .16353 L
.33919 .12231 L
.3417 .12231 L
.34421 .10862 L
.34648 .09577 L
.34853 .0851 L
.35038 .07619 L
.35208 .06868 L
.35365 .06231 L
.3551 .05691 L
.35646 .05234 L
.35773 .04853 L
.35894 .04541 L
.3601 .04298 L
.36121 .04125 L
.36229 .04032 L
.36336 .04036 L
.36442 .04178 L
.36551 .04556 L
.36667 .05578 L
.36801 .03667 L
.36901 .03617 L
.37 .03149 L
.37091 .02966 L
.37178 .03172 L
.3727 .02751 L
.37354 .0204 L
.37425 .0204 L
.37496 .01752 L
.37562 .01588 L
.37626 .01509 L
.37688 .01479 L
Mistroke
.37749 .01487 L
.37811 .01522 L
.37873 .01583 L
.37936 .01665 L
.38001 .0177 L
.38067 .01896 L
.38136 .02043 L
.38207 .02214 L
.38282 .02409 L
.38359 .0263 L
.38441 .02879 L
.38527 .03161 L
.38618 .03476 L
.38715 .03828 L
.38818 .0422 L
.38928 .04656 L
.39045 .05139 L
.39171 .05673 L
.39307 .06266 L
.39453 .06929 L
.3961 .07683 L
.39781 .08569 L
.39968 .0967 L
.40174 .11011 L
.40403 .12312 L
.40656 .14146 L
.40941 .15961 L
.41258 .14486 L
.41549 .14396 L
.41839 .14613 L
.42132 .15057 L
.42433 .15688 L
.42746 .16503 L
.43072 .17518 L
.43417 .18759 L
.43784 .20264 L
.44177 .22061 L
.44601 .24135 L
.45063 .2632 L
.45563 .28184 L
.46096 .26372 L
.46597 .24282 L
.47061 .22794 L
.47499 .21829 L
.47919 .21236 L
.4833 .20919 L
.48734 .20816 L
.49137 .20894 L
.49541 .21131 L
.4995 .21516 L
Mistroke
.50365 .22045 L
.50789 .22718 L
.51226 .23542 L
.51677 .24528 L
.52145 .25695 L
.52634 .27065 L
.53147 .28671 L
.53689 .3056 L
.54264 .32798 L
.54878 .35496 L
.5554 .38838 L
.56261 .43242 L
.5706 .49991 L
.57978 .60053 L
.59074 .51301 L
.60015 .51301 L
.60956 .48827 L
.61854 .47369 L
.62725 .47348 L
.63597 .49377 L
.64504 .44699 L
.65328 .3689 L
.66015 .30534 L
.6659 .24288 L
.67054 .2028 L
.67447 .17223 L
.67787 .1488 L
.68085 .1488 L
.68383 .13016 L
.68648 .11307 L
.68883 .09926 L
.69093 .08796 L
.69284 .07857 L
.69458 .07068 L
.69618 .064 L
.69766 .05833 L
.69904 .05352 L
.70034 .04949 L
.70157 .04617 L
.70274 .04353 L
.70386 .04158 L
.70495 .04038 L
.70601 .04009 L
.70707 .04101 L
.70815 .04385 L
.70928 .05097 L
.71053 .04097 L
.71161 .03947 L
.71266 .03286 L
.71359 .0298 L
Mistroke
.71447 .03034 L
.71536 .039 L
.7164 .02081 L
.71712 .02081 L
.71784 .01773 L
.7185 .01594 L
.71914 .01506 L
.71976 .01472 L
.72037 .01474 L
.72098 .01506 L
.7216 .01563 L
.72223 .01643 L
.72287 .01744 L
.72353 .01866 L
.72422 .0201 L
.72492 .02177 L
.72566 .02367 L
.72643 .02584 L
.72724 .02828 L
.72809 .03103 L
.72899 .03412 L
.72995 .03757 L
.73097 .04142 L
.73205 .04569 L
.73321 .05042 L
.73445 .05565 L
.73579 .06146 L
.73723 .06793 L
.73878 .07525 L
.74046 .08379 L
.74229 .09424 L
.74431 .1079 L
.74657 .12021 L
.74904 .13712 L
.75182 .16523 L
.75509 .14547 L
.75801 .14389 L
.7609 .14558 L
.76383 .14967 L
.76682 .15567 L
.76992 .1635 L
.77316 .17328 L
.77657 .18528 L
.7802 .19984 L
.78408 .21731 L
.78827 .23762 L
.79282 .25951 L
.79775 .27912 L
.80304 .26752 L
.80811 .24615 L
Mistroke
.81281 .23016 L
.81723 .2197 L
.82146 .2132 L
.82558 .20958 L
.82963 .2082 L
.83366 .20868 L
.8377 .21077 L
.84177 .21436 L
.84591 .21939 L
.85014 .22586 L
.85448 .23382 L
.85896 .24339 L
.86361 .25471 L
.86846 .26803 L
.87355 .28364 L
.87891 .30198 L
.88459 .32367 L
.89066 .34972 L
.89719 .38178 L
.90428 .42339 L
.91211 .48468 L
.92102 .60332 L
.93203 .58816 L
.94277 .52097 L
.95232 .48867 L
.9613 .47392 L
.97002 .32932 L
.97619 .32932 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[69]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`006`3oool00`000000oooo0?ooo`3o0?ooo`<0oooo000K0?ooo`030000003oool0oooo0?l0
oooo0`3oool001/0oooo0P00003o0?ooo`@0oooo000K0?ooo`030000003oool0oooo0?l0oooo0`3o
ool001/0oooo00<000000?ooo`3oool0FP3oool2000005T0oooo00<000000?ooo`3oool0BP3oool0
01/0oooo00<000000?ooo`3oool0FP3oool2000005P0oooo00<000000?ooo`000000B`3oool001/0
oooo00<000000?ooo`3oool0F@3oool4000005L0oooo00<000000?ooo`000000B`3oool001/0oooo
00<000000?ooo`3oool0F@3oool010000000oooo0?ooo`00001G0?ooo`040000003oool0oooo0000
04X0oooo000K0?ooo`030000003oool0oooo05T0oooo00@000000?ooo`3oool00000EP3oool20000
0080oooo00<000000?ooo`3oool0B03oool001/0oooo0P00001I0?ooo`8000000P3oool2000005D0
oooo0P0000020?ooo`030000003oool0oooo04P0oooo000K0?ooo`030000003oool0oooo05P0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`1C0?ooo`8000000`3oool00`000000oooo
0?ooo`170?ooo`006`3oool00`000000oooo0?ooo`1F0?ooo`<00000103oool00`000000oooo0?oo
o`1A0?ooo`@000000`3oool00`000000oooo0?ooo`170?ooo`006`3oool00`000000oooo0?ooo`1F
0?ooo`<00000103oool00`000000oooo0?ooo`1@0?ooo`040000003oool00000000000@0oooo00<0
00000?ooo`3oool0A`3oool001/0oooo00<000000?ooo`3oool0E@3oool010000000oooo00000000
00050?ooo`030000003oool0oooo04l0oooo00<000000?ooo`0000001@3oool00`000000oooo0?oo
o`170?ooo`006`3oool00`000000oooo0?ooo`1E0?ooo`040000003oool0oooo000000D0oooo00<0
00000?ooo`3oool0C`3oool00`000000oooo000000060?ooo`030000003oool0oooo04H0oooo000K
0?ooo`030000003oool0oooo05@0oooo00D000000?ooo`3oool0oooo000000050?ooo`030000003o
ool0oooo04l0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`160?ooo`006`3oool0
0`000000oooo0?ooo`1D0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0CP3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo04H0oooo000K0?ooo`800000E@3oool00`00
0000oooo0?ooo`080?ooo`030000003oool0oooo04d0oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`160?ooo`006`3oool00`000000oooo0?ooo`1D0?ooo`030000003oool0oooo00P0
oooo00<000000?ooo`3oool0C@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo04D0
oooo000K0?ooo`030000003oool0oooo05@0oooo00<000000?ooo`3oool0203oool00`000000oooo
0?ooo`1<0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0A@3oool001/0oooo00<0
00000?ooo`3oool0D`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo04`0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`150?ooo`006`3oool00`000000oooo0?ooo`1C
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0C03oool00`000000oooo0?ooo`09
0?ooo`030000003oool0oooo04D0oooo000K0?ooo`030000003oool0oooo05<0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`1<0?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool0A@3oool001/0oooo00<000000?ooo`3oool0D`3oool00`000000oooo0?ooo`0:0?ooo`03
0000003oool0oooo04X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`150?ooo`00
6`3oool2000000<0oooo00<000000?ooo`3oool0CP3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo04X0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`140?ooo`006`3o
ool010000000oooo0?ooo`3oool2000004l0oooo00<000000?ooo`3oool02`3oool00`000000oooo
0?ooo`1:0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0A03oool001/0oooo00L0
00000?ooo`3oool0oooo0000003oool0000004h0oooo00<000000?ooo`3oool02`3oool00`000000
oooo0?ooo`1:0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0A03oool001/0oooo
00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`1<0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool0BP3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo04@0
oooo000K0?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0C03oool00`000000
oooo0?ooo`0<0?ooo`030000003oool0oooo04T0oooo00<000000?ooo`3oool02`3oool00`000000
oooo0?ooo`140?ooo`000P3oool2000000@0oooo00<000000?ooo`3oool00`3oool2000000<0oooo
0`0000050?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0C03oool00`000000
oooo0?ooo`0<0?ooo`030000003oool0oooo04P0oooo00<000000?ooo`3oool0303oool00`000000
oooo0?ooo`140?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo0000
00D0oooo00<000000?ooo`3oool00P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0
oooo04`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`180?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0A03oool000050?ooo`000000oooo0?ooo`000000203oool0
10000000oooo0?ooo`0000050?ooo`030000003oool0oooo0080oooo0`0000030?ooo`030000003o
ool0oooo04`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`180?ooo`030000003o
ool0oooo00d0oooo00<000000?ooo`3oool0@`3oool000050?ooo`000000oooo0?ooo`000000203o
ool010000000oooo0?ooo`0000020?ooo`<000001@3oool00`000000oooo000000040?ooo`030000
003oool0oooo04X0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`180?ooo`030000
003oool0oooo00d0oooo00<000000?ooo`3oool0@`3oool000050?ooo`000000oooo0?ooo`000000
203oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo00D0oooo0P0000050?ooo`03
0000003oool0oooo04T0oooo0P00000?0?ooo`030000003oool0oooo04L0oooo00<000000?ooo`3o
ool03P3oool00`000000oooo0?ooo`130?ooo`0000D0oooo0000003oool0oooo000000080?ooo`04
0000003oool0oooo00000080oooo00<000000?ooo`3oool01@3oool2000000D0oooo00D000000?oo
o`3oool0oooo000000170?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0A`3oool0
0`000000oooo0?ooo`0>0?ooo`030000003oool0oooo04<0oooo00020?ooo`8000002P3oool20000
00<0oooo100000040?ooo`8000001@3oool010000000oooo0?ooo`3oool2000004H0oooo00<00000
0?ooo`3oool03P3oool00`000000oooo0?ooo`170?ooo`030000003oool0oooo00l0oooo00<00000
0?ooo`3oool0@P3oool001/0oooo0P0000050?ooo`040000003oool0oooo00000080oooo00<00000
0?ooo`3oool0@`3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo04H0oooo00<00000
0?ooo`3oool03`3oool00`000000oooo0?ooo`120?ooo`006`3oool2000000H0oooo0P0000030?oo
o`030000003oool0oooo04<0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`160?oo
o`030000003oool0oooo00l0oooo00<000000?ooo`3oool0@P3oool001/0oooo0P0000060?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool0@P3oool00`000000oooo0?ooo`0?0?ooo`03
0000003oool0000004H0oooo00<000000?ooo`3oool03`3oool00`000000oooo000000120?ooo`00
6`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0440oooo00<000000?ooo`3oool0
403oool010000000oooo0000000000150?ooo`030000003oool0oooo00l0oooo0P0000000`3oool0
00000?ooo`100?ooo`006`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0400oooo
00<000000?ooo`3oool0403oool01@000000oooo0000003oool0000004<0oooo0P00000A0?ooo`80
00000P3oool00`000000oooo0?ooo`0n0?ooo`006`3oool00`000000oooo0?ooo`0<0?ooo`030000
003oool0oooo0400oooo00<000000?ooo`3oool04@3oool010000000oooo0?ooo`0000130?ooo`03
0000003oool0oooo0100oooo0P0000020?ooo`030000003oool0oooo03h0oooo000K0?ooo`030000
003oool0oooo00d0oooo00<000000?ooo`3oool0?`3oool00`000000oooo0?ooo`0A0?ooo`050000
003oool0oooo0?ooo`000000@P3oool00`000000oooo0?ooo`0@0?ooo`050000003oool0oooo0?oo
o`000000@03oool001/0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0o0?ooo`03
0000003oool0oooo01D0oooo00<000000?ooo`3oool0@03oool00`000000oooo0?ooo`0@0?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool0?@3oool001/0oooo00<000000?ooo`3oool0
3P3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0
?P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
?@3oool001/0oooo0P00000?0?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool05P3o
ool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0?@3o
ool001/0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0
oooo01L0oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0
oooo03`0oooo000K0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0?P3oool00`00
0000oooo0?ooo`0G0?ooo`030000003oool0oooo03d0oooo00<000000?ooo`3oool05`3oool00`00
0000oooo0?ooo`0l0?ooo`006`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo03d0
oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0m0?ooo`030000003oool0oooo01P0
oooo00<000000?ooo`3oool0>`3oool001/0oooo00<000000?ooo`3oool03P3oool00`000000oooo
0?ooo`0m0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0?@3oool00`000000oooo
0?ooo`0H0?ooo`030000003oool0oooo03/0oooo000K0?ooo`030000003oool0oooo00l0oooo00<0
00000?ooo`3oool0?03oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo03d0oooo00<0
00000?ooo`3oool0603oool00`000000oooo0?ooo`0k0?ooo`006`3oool00`000000oooo0?ooo`0?
0?ooo`030000003oool0oooo03`0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0l
0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0>`3oool001/0oooo0P00000@0?oo
o`030000003oool0oooo03`0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0l0?oo
o`030000003oool0oooo01X0oooo00<000000?ooo`3oool0>P3oool001/0oooo00<000000?ooo`3o
ool03`3oool00`000000oooo0?ooo`0l0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3o
ool0>`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo03X0oooo000K0?ooo`030000
003oool0oooo0100oooo00<000000?ooo`3oool0>P3oool00`000000oooo0?ooo`0J0?ooo`030000
003oool0oooo03/0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`0j0?ooo`006`3o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool06P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3oool06P3o
ool00`000000oooo0?ooo`0:0?ooo`800000;P3oool001/0oooo00<000000?ooo`3oool0403oool0
0`000000oooo0?ooo`0W0?ooo`<00000403oool00`000000oooo0?ooo`0J0?ooo`030000003oool0
oooo00P0oooo1P0000030?ooo`800000:03oool00`000000oooo0?ooo`0;0?ooo`@000000P3oool3
000000L0oooo00<000000?ooo`3oool0203oool010000000oooo0?ooo`00000Q0?ooo`@000000`3o
ool2000000<0oooo000K0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0:P3oool0
0`000000oooo0?ooo`0=0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool01`3oool0
2@000000oooo0?ooo`3oool000000?ooo`000000oooo000000020?ooo`030000003oool0oooo02D0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01h0
oooo00<000000?ooo`3oool00`3oool010000000oooo0?ooo`0000020?ooo`006`3oool00`000000
oooo0?ooo`0A0?ooo`030000003oool0oooo00T0oooo0`00000M0?ooo`030000003oool0oooo00`0
oooo00<000000?ooo`3oool0703oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo0080
oooo00D000000?ooo`000000oooo000000020?ooo`030000003oool0oooo02D0oooo00<000000?oo
o`3oool03@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool01P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01h0oooo00<000000?oo
o`3oool00P3oool010000000oooo0?ooo`0000020?ooo`006`3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00P0oooo00D000000?ooo`3oool0oooo0000000I0?ooo`<000003`3oool0
0`000000oooo0?ooo`0L0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0200000080oooo00<000000?ooo`3oool09@3oool00`000000oooo0?ooo`0=
0?ooo`040000003oool0oooo0?ooo`<000001`3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool07P3oool01@000000oooo0?ooo`3oool000000080oooo
0@0000010?ooo`40oooo000K0?ooo`8000004P3oool00`000000oooo0?ooo`080?ooo`030000003o
ool0oooo0080oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0?0?ooo`030000003o
ool0oooo01`0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`030?ooo`050000003o
ool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0T0?ooo`030000003oool0oooo00h0oooo
00D000000?ooo`3oool0oooo000000090?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0N0?ooo`040000003oool0oooo00000080oooo0@0000010?oo
o`40oooo000K0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool01P3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool03`3oool00`000000
oooo0?ooo`0M0?ooo`030000003oool0oooo00@0oooo00@000000?ooo`3oool0oooo0`0000030?oo
o`040000003oool0oooo000002H0oooo00<000000?ooo`3oool0303oool3000000<0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00L0oooo00<00000
0?ooo`3oool06`3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo
000K0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool01P3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo01@0oooo1000000>0?ooo`030000003oool0oooo01d0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo0P00
000W0?ooo`030000003oool0oooo00h0oooo00@000000?ooo`3oool0oooo100000070?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0K0?ooo`800000
103oool2000000<0oooo000K0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo02H0oooo00<000000?ooo`3oool07P3o
ool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool09P3o
ool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool02@3o
ool00`000000oooo0?ooo`0V0?ooo`006`3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0
oooo00@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0U0?ooo`030000003oool0
oooo01h0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo02H0oooo00<000000?ooo`3oool07@3oool00`000000oooo0?ooo`030?ooo`030000003oool0
oooo00X0oooo00<000000?ooo`3oool09@3oool001/0oooo00<000000?ooo`3oool04`3oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool09@3oool00`00
0000oooo0?ooo`0N0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0U0?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo02D0oooo000K0?ooo`030000003oool0oooo01<0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo02@0
oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool0903oool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo0080
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0U0?ooo`005@3ooooo000000X00000
0P3oool001/0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`070?ooo`060000003o
ool0oooo0000003oool000002P3oool010000000oooo0?ooo`0000090?ooo`030000003oool0oooo
00T0oooo00<000000?ooo`3oool02P3oool00`000000oooo0000000:0?ooo`030000003oool0oooo
00T0oooo00<000000?ooo`3oool02@3oool2000000030?ooo`000000oooo00T0oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`050000003oool0oooo0?ooo`0000001`3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool02P3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00@0oooo000K0?ooo`030000003oool0oooo
01@0oooo00D000000?ooo`3oool0oooo0000000=0?ooo`030000003oool0oooo0280oooo00<00000
0?ooo`3oool0803oool010000000oooo0?ooo`00000?0?ooo`030000003oool0oooo02<0oooo00<0
00000?ooo`3oool07P3oool010000000oooo0?ooo`00000?0?ooo`030000003oool0oooo02@0oooo
000K0?ooo`030000003oool0oooo01@0oooo00D000000?ooo`3oool0oooo0000000=0?ooo`030000
003oool0oooo0280oooo00<000000?ooo`3oool0803oool010000000oooo0?ooo`00000?0?ooo`03
0000003oool0oooo0280oooo00<000000?ooo`3oool07`3oool010000000oooo0?ooo`00000@0?oo
o`030000003oool0oooo02<0oooo000K0?ooo`030000003oool0oooo01@0oooo00D000000?ooo`3o
ool0oooo0000000>0?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool0803oool01000
0000oooo0?ooo`00000@0?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool0803oool0
0`000000oooo0000000@0?ooo`030000003oool0oooo02<0oooo000K0?ooo`030000003oool0oooo
01@0oooo00D000000?ooo`3oool0oooo0000000>0?ooo`030000003oool0oooo0240oooo00<00000
0?ooo`3oool0803oool00`000000oooo0000000A0?ooo`030000003oool0oooo0240oooo00<00000
0?ooo`3oool0803oool00`000000oooo0000000@0?ooo`030000003oool0oooo02<0oooo000K0?oo
o`030000003oool0oooo01@0oooo00D000000?ooo`3oool0oooo0000000>0?ooo`030000003oool0
oooo0240oooo00<000000?ooo`3oool08@3oool200000140oooo00<000000?ooo`3oool08@3oool0
0`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool08P3oool0
01/0oooo0P00000E0?ooo`050000003oool0oooo0?ooo`0000003`3oool00`000000oooo0?ooo`0P
0?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0Q
0?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0R
0?ooo`006`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool000000100oooo00<000000?oo
o`3oool0803oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo0140oooo00<000000?oo
o`3oool0803oool00`000000oooo0?ooo`0e0?ooo`030000003oool0oooo0240oooo000K0?ooo`03
0000003oool0oooo01D0oooo00<000000?ooo`0000004@3oool00`000000oooo0?ooo`0O0?ooo`03
0000003oool0oooo03D0oooo00<000000?ooo`3oool0803oool00`000000oooo0?ooo`0e0?ooo`03
0000003oool0oooo0240oooo000K0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`000000
4@3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo03D0oooo00<000000?ooo`3oool0
803oool00`000000oooo0?ooo`0e0?ooo`030000003oool0oooo0240oooo000K0?ooo`030000003o
ool0oooo01H0oooo0P00000A0?ooo`030000003oool0oooo01l0oooo00<000000?ooo`3oool0=P3o
ool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo03L0oooo00<000000?ooo`3oool0803o
ool001/0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0
oooo03H0oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0g0?ooo`030000003oool0
oooo0200oooo000K0?ooo`030000003oool0oooo02X0oooo00<000000?ooo`3oool07@3oool00`00
0000oooo0?ooo`0h0?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3oool0=`3oool00`00
0000oooo0?ooo`0P0?ooo`006`3oool2000002/0oooo00<000000?ooo`3oool07@3oool00`000000
oooo0?ooo`0h0?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3oool0=`3oool00`000000
oooo0?ooo`0P0?ooo`006`3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01d0oooo
00<000000?ooo`3oool0>03oool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo03P0oooo
00<000000?ooo`3oool07`3oool001/0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?oo
o`0M0?ooo`030000003oool0oooo03P0oooo00<000000?ooo`3oool07@3oool00`000000oooo0?oo
o`0h0?ooo`030000003oool0oooo01l0oooo000K0?ooo`030000003oool0oooo02/0oooo00<00000
0?ooo`3oool0703oool00`000000oooo0?ooo`0h0?ooo`030000003oool0oooo01d0oooo00<00000
0?ooo`3oool0>03oool00`000000oooo0?ooo`0O0?ooo`006`3oool00`000000oooo0?ooo`0[0?oo
o`030000003oool0oooo01`0oooo00<000000?ooo`3oool0>@3oool00`000000oooo0?ooo`0L0?oo
o`030000003oool0oooo03T0oooo00<000000?ooo`3oool07P3oool001/0oooo00<000000?ooo`3o
ool0:`3oool00`000000oooo0?ooo`0L0?ooo`030000003oool0oooo03T0oooo00<000000?ooo`3o
ool06`3oool00`000000oooo0?ooo`0j0?ooo`030000003oool0oooo01h0oooo000K0?ooo`030000
003oool0oooo02`0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0i0?ooo`030000
003oool0oooo01/0oooo00<000000?ooo`3oool0>P3oool00`000000oooo0?ooo`0C0?ooo`<00000
203oool001/0oooo0P00000]0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0>P3o
ool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool04`3o
ool00`000000oooo0?ooo`080?ooo`006`3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0
oooo01X0oooo00<000000?ooo`3oool0>`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0
oooo03X0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`006`3oool00`00
0000oooo0?ooo`0/0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0>`3oool00`00
0000oooo0?ooo`0J0?ooo`030000003oool0oooo03/0oooo00<000000?ooo`3oool04P3oool00`00
0000oooo0?ooo`080?ooo`006`3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo01X0
oooo00<000000?ooo`3oool0>`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo03/0
oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`080?ooo`006`3oool00`000000oooo
0?ooo`0]0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0>`3oool00`000000oooo
0?ooo`0J0?ooo`030000003oool0oooo03/0oooo00<000000?ooo`3oool04P3oool00`000000oooo
0?ooo`080?ooo`006`3oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo01T0oooo00<0
00000?ooo`3oool0?03oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo03/0oooo00<0
00000?ooo`3oool04P3oool00`000000oooo0?ooo`080?ooo`006`3oool00`000000oooo0?ooo`0]
0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0?03oool00`000000oooo0?ooo`0H
0?ooo`030000003oool0oooo03`0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`08
0?ooo`006`3oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo01T0oooo00<000000?oo
o`3oool0?03oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo03`0oooo00<000000?oo
o`3oool04P3oool00`000000oooo0?ooo`080?ooo`006`3oool2000002h0oooo00<000000?ooo`3o
ool06@3oool00`000000oooo0?ooo`0l0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3o
ool0?@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00P0oooo000K0?ooo`030000
003oool0oooo02d0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0n0?ooo`030000
003oool0oooo01L0oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0@0?ooo`030000
003oool0oooo00T0oooo000K0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3oool05`3o
ool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0?@3o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00T0oooo000K0?ooo`030000003oool0
oooo02h0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0
oooo01L0oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0
oooo00T0oooo000K0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3oool05`3oool00`00
0000oooo0?ooo`0n0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0?P3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo00T0oooo00020?ooo`800000103oool00`000000
oooo0?ooo`020?ooo`@000000P3oool3000000D0oooo00<000000?ooo`3oool0;P3oool00`000000
oooo0?ooo`0G0?ooo`030000003oool0oooo03l0oooo00<000000?ooo`3oool05P3oool00`000000
oooo0?ooo`0n0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool02@3oool000050?oo
o`000000oooo0?ooo`0000002P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0080
oooo00<000000?ooo`3oool0;`3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo03l0
oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo00l0
oooo00<000000?ooo`3oool02@3oool000050?ooo`000000oooo0?ooo`0000002P3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo0080oooo0P00000`0?ooo`030000003oool0oooo01H0
oooo00<000000?ooo`3oool0?`3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo03h0
oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`090?ooo`0000D0oooo0000003oool0
oooo0000000:0?ooo`040000003oool0oooo0?ooo`<000001@3oool00`000000oooo0?ooo`0_0?oo
o`030000003oool0oooo01H0oooo00<000000?ooo`3oool0?`3oool00`000000oooo0?ooo`0E0?oo
o`030000003oool0oooo0400oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`090?oo
o`0000D0oooo0000003oool0oooo0000000:0?ooo`050000003oool0oooo0?ooo`0000001`3oool0
0`000000oooo0?ooo`0_0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0?`3oool0
0`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0400oooo00<000000?ooo`3oool03P3oool0
0`000000oooo0?ooo`090?ooo`0000D0oooo0000003oool0oooo000000080?ooo`<000000`3oool0
0`000000oooo0?ooo`050?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool05P3oool0
0`000000oooo0?ooo`0o0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0@03oool0
0`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00T0oooo00020?ooo`8000002`3oool01000
0000oooo0?ooo`3oool4000000@0oooo00<000000?ooo`3oool0;`3oool00`000000oooo0?ooo`0E
0?ooo`030000003oool0oooo0400oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`10
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool02@3oool001/0oooo00<000000?oo
o`3oool0<03oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo0400oooo00<000000?oo
o`3oool05@3oool00`000000oooo0?ooo`100?ooo`030000003oool0oooo00h0oooo00<000000?oo
o`3oool02@3oool001/0oooo00<000000?ooo`3oool0<03oool00`000000oooo0?ooo`0D0?ooo`03
0000003oool0oooo0400oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`100?ooo`03
0000003oool0oooo00h0oooo00<000000?ooo`3oool02@3oool001/0oooo0P00000a0?ooo`030000
003oool0oooo01@0oooo00<000000?ooo`3oool0@@3oool00`000000oooo0?ooo`0D0?ooo`030000
003oool0oooo0440oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`006`3o
ool00`000000oooo0?ooo`0`0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0@@3o
ool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo0440oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`090?ooo`006`3oool00`000000oooo0?ooo`0`0?ooo`030000003oool0
oooo01@0oooo00<000000?ooo`3oool0@@3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0
oooo0440oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`006`3oool00`00
0000oooo0?ooo`0`0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0@@3oool00`00
0000oooo0?ooo`0D0?ooo`030000003oool0oooo0440oooo00<000000?ooo`3oool03@3oool00`00
0000oooo0?ooo`090?ooo`006`3oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo01<0
oooo00<000000?ooo`3oool0@@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo0480
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`006`3oool00`000000oooo
0?ooo`0a0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool0@@3oool00`000000oooo
0?ooo`0C0?ooo`030000003oool0oooo0480oooo00<000000?ooo`3oool03@3oool00`000000oooo
0?ooo`090?ooo`006`3oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo01<0oooo00<0
00000?ooo`3oool0@@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo0480oooo00<0
00000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`006`3oool200000380oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`110?ooo`030000003oool0oooo01<0oooo00<00000
0?ooo`3oool0@`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00X0oooo000K0?oo
o`030000003oool0oooo0340oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`120?oo
o`030000003oool0oooo01<0oooo00<000000?ooo`3oool0@`3oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo00X0oooo000K0?ooo`030000003oool0oooo0340oooo00<000000?ooo`3o
ool04P3oool00`000000oooo0?ooo`130?ooo`030000003oool0oooo0180oooo00<000000?ooo`3o
ool0@`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00X0oooo000K0?ooo`030000
003oool0oooo0340oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`130?ooo`030000
003oool0oooo0180oooo00<000000?ooo`3oool0@`3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo00X0oooo000K0?ooo`030000003oool0oooo0340oooo00<000000?ooo`3oool04P3o
ool00`000000oooo0?ooo`130?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0@`3o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00X0oooo000K0?ooo`030000003oool0
oooo0340oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`130?ooo`030000003oool0
oooo0180oooo00<000000?ooo`3oool0@`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo00X0oooo000K0?ooo`030000003oool0oooo0380oooo00<000000?ooo`3oool04@3oool00`00
0000oooo0?ooo`130?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0A03oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo00X0oooo000K0?ooo`800000<`3oool00`000000
oooo0?ooo`0A0?ooo`030000003oool0oooo04<0oooo00<000000?ooo`3oool04@3oool00`000000
oooo0?ooo`140?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool02P3oool001/0oooo
00<000000?ooo`3oool0<P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool0A03oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo04D0oooo
00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`006`3oool00`000000oooo0?oo
o`0b0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`000000103oool00`000000oooo0?oo
o`140?ooo`030000003oool0oooo00X0oooo100000020?ooo`030000003oool0oooo04D0oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`006`3oool00`000000oooo0?ooo`0b
0?ooo`030000003oool0oooo00X0oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?oo
o`140?ooo`030000003oool0oooo00T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000
003oool0oooo04D0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`006`3o
ool00`000000oooo0?ooo`0b0?ooo`030000003oool0oooo00T0oooo00D000000?ooo`3oool0oooo
000000030?ooo`030000003oool0oooo04@0oooo00<000000?ooo`3oool02@3oool00`000000oooo
0?ooo`020?ooo`030000003oool0000004L0oooo00<000000?ooo`3oool02@3oool00`000000oooo
0?ooo`0;0?ooo`006`3oool00`000000oooo0?ooo`0b0?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool00P3oool010000000oooo0?ooo`0000160?ooo`030000003oool0oooo00P0oooo
00<000000?ooo`3oool00`3oool2000004P0oooo00<000000?ooo`3oool02@3oool00`000000oooo
0?ooo`0;0?ooo`006`3oool00`000000oooo0?ooo`0b0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo000000170?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`160?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0303oool001/0oooo0P00000c0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool00`3oool00`000000oooo000000180?ooo`030000003oool0oooo00H0oooo00<00000
0?ooo`3oool01@3oool00`000000oooo0?ooo`160?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool0303oool001/0oooo00<000000?ooo`3oool0<P3oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo00D0oooo0P0000180?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool0CP3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00d0oooo000K0?ooo`03
0000003oool0oooo03<0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`050?ooo`80
0000B03oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo04l0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0=0?ooo`006`3oool00`000000oooo0?ooo`0c0?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`160?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`3oool0C`3oool00`000000oooo0?ooo`070?ooo`030000
003oool0oooo00d0oooo000K0?ooo`030000003oool0oooo03<0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo04H0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`1@0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool03P3o
ool001/0oooo00<000000?ooo`3oool0<`3oool00`000000oooo0?ooo`050?ooo`030000003oool0
oooo0500oooo00<000000?ooo`3oool00P3oool300000580oooo00<000000?ooo`3oool01P3oool0
0`000000oooo0?ooo`0>0?ooo`00203oool2000000@0oooo00<000000?ooo`3oool00P3oool40000
00@0oooo00<000000?ooo`3oool0<`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
0500oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`1C0?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool03`3oool000L0oooo00@000000?ooo`3oool00000203oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo03<0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`1A0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0D`3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00l0oooo00070?ooo`040000003oool0oooo0000
00T0oooo00<000000?ooo`3oool0103oool2000003@0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`1A0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0D`3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00l0oooo00070?ooo`040000003oool0oooo0000
00X0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0c0?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool0DP3oool010000000oooo0?ooo`00001F0?ooo`030000003oool0
oooo00@0oooo00<000000?ooo`3oool03`3oool000L0oooo00@000000?ooo`3oool000002`3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0oooo03@0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`1B0?ooo`040000003oool0oooo000005H0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`0?0?ooo`001`3oool010000000oooo0?ooo`0000080?ooo`040000003o
ool0oooo000000@0oooo00<000000?ooo`3oool0=03oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo0580oooo00@000000?ooo`3oool00000EP3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo0100oooo00080?ooo`8000002P3oool2000000D0oooo00<000000?ooo`3oool0
=03oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0580oooo00@000000?ooo`3oool0
0000EP3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0100oooo000K0?ooo`030000
003oool0oooo03@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1B0?ooo`040000
003oool0oooo000005H0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0@0?ooo`00
6`3oool00`000000oooo0?ooo`0d0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
DP3oool010000000oooo0?ooo`00001F0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool0403oool001/0oooo0P00000e0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
D`3oool010000000oooo0?ooo`00001F0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool0403oool001/0oooo00<000000?ooo`3oool0=@3oool01@000000oooo0?ooo`3oool0000005D0
oooo00@000000?ooo`3oool00000EP3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
0100oooo000K0?ooo`030000003oool0oooo03D0oooo00D000000?ooo`3oool0oooo0000001E0?oo
o`030000003oool0000005L0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0A0?oo
o`006`3oool00`000000oooo0?ooo`0e0?ooo`050000003oool0oooo0?ooo`000000E@3oool00`00
0000oooo0000001G0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool04@3oool001/0
oooo00<000000?ooo`3oool0=@3oool01@000000oooo0?ooo`3oool0000005D0oooo00<000000?oo
o`000000E`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0140oooo000K0?ooo`03
0000003oool0oooo03D0oooo00@000000?ooo`3oool00000EP3oool00`000000oooo0000001G0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool04@3oool001/0oooo00<000000?ooo`3o
ool0=@3oool010000000oooo0?ooo`00001F0?ooo`030000003oool0000005L0oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`0A0?ooo`006`3oool2000003L0oooo0P00001H0?ooo`80
0000F03oool01@000000oooo0?ooo`3oool0000001<0oooo000K0?ooo`030000003oool0oooo03H0
oooo0P00001H0?ooo`800000F03oool010000000oooo0?ooo`00000D0?ooo`006`3oool00`000000
oooo0?ooo`0f0?ooo`030000003oool0oooo05L0oooo0P00001H0?ooo`040000003oool0oooo0000
01@0oooo000K0?ooo`030000003oool0oooo03H0oooo00<000000?ooo`3oool0E`3oool00`000000
oooo0?ooo`1G0?ooo`040000003oool0oooo000001@0oooo000K0?ooo`030000003oool0oooo0900
oooo00<000000?ooo`3oool0E`3oool00`000000oooo0000000E0?ooo`006`3oool00`000000oooo
0?ooo`2@0?ooo`030000003oool0oooo05L0oooo00<000000?ooo`0000005@3oool001/0oooo00<0
00000?ooo`3oool0T03oool00`000000oooo0?ooo`1G0?ooo`8000005P3oool001/0oooo0P00003[
0?ooo`030000003oool0oooo01D0oooo000K0?ooo`030000003oool0oooo0?l0oooo0`3oool001/0
oooo00<000000?ooo`3oool0o`3oool30?ooo`006`3oool00`000000oooo0?ooo`3o0?ooo`<0oooo
000K0?ooo`030000003oool0oooo0?l0oooo0`3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0
oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-2.15299, -0.00194469, \
0.0788368, 0.00139208}}]
}, Open  ]],

Cell["\<\
Using the PI step controller the step sizes are slightly smaller.\
\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(pisol\  = \ 
      NDSolve[system, \ Method \[Rule] PIERK]; \), "\[IndentingNewLine]", 
    \(StepDataPlot[pisol]; \)}], "Input",
  CellLabel->"In[71]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0479504 -0.0282638 4.0136 [
[.26356 .3606 -3 -9 ]
[.26356 .3606 3 0 ]
[.50331 .3606 -6 -9 ]
[.50331 .3606 6 0 ]
[.74307 .3606 -6 -9 ]
[.74307 .3606 6 0 ]
[.98282 .3606 -6 -9 ]
[.98282 .3606 6 0 ]
[.01131 .05201 -24 -4.5 ]
[.01131 .05201 0 4.5 ]
[.01131 .13228 -24 -4.5 ]
[.01131 .13228 0 4.5 ]
[.01131 .21255 -24 -4.5 ]
[.01131 .21255 0 4.5 ]
[.01131 .29282 -24 -4.5 ]
[.01131 .29282 0 4.5 ]
[.01131 .45337 -24 -4.5 ]
[.01131 .45337 0 4.5 ]
[.01131 .53364 -24 -4.5 ]
[.01131 .53364 0 4.5 ]
[.01131 .61391 -24 -4.5 ]
[.01131 .61391 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.26356 .3731 m
.26356 .37935 L
s
[(5)] .26356 .3606 0 1 Mshowa
.50331 .3731 m
.50331 .37935 L
s
[(10)] .50331 .3606 0 1 Mshowa
.74307 .3731 m
.74307 .37935 L
s
[(15)] .74307 .3606 0 1 Mshowa
.98282 .3731 m
.98282 .37935 L
s
[(20)] .98282 .3606 0 1 Mshowa
.125 Mabswid
.07176 .3731 m
.07176 .37685 L
s
.11971 .3731 m
.11971 .37685 L
s
.16766 .3731 m
.16766 .37685 L
s
.21561 .3731 m
.21561 .37685 L
s
.31151 .3731 m
.31151 .37685 L
s
.35946 .3731 m
.35946 .37685 L
s
.40741 .3731 m
.40741 .37685 L
s
.45536 .3731 m
.45536 .37685 L
s
.55126 .3731 m
.55126 .37685 L
s
.59921 .3731 m
.59921 .37685 L
s
.64716 .3731 m
.64716 .37685 L
s
.69512 .3731 m
.69512 .37685 L
s
.79102 .3731 m
.79102 .37685 L
s
.83897 .3731 m
.83897 .37685 L
s
.88692 .3731 m
.88692 .37685 L
s
.93487 .3731 m
.93487 .37685 L
s
.25 Mabswid
0 .3731 m
1 .3731 L
s
.02381 .05201 m
.03006 .05201 L
s
[(0.02)] .01131 .05201 1 0 Mshowa
.02381 .13228 m
.03006 .13228 L
s
[(0.04)] .01131 .13228 1 0 Mshowa
.02381 .21255 m
.03006 .21255 L
s
[(0.06)] .01131 .21255 1 0 Mshowa
.02381 .29282 m
.03006 .29282 L
s
[(0.08)] .01131 .29282 1 0 Mshowa
.02381 .45337 m
.03006 .45337 L
s
[(0.12)] .01131 .45337 1 0 Mshowa
.02381 .53364 m
.03006 .53364 L
s
[(0.14)] .01131 .53364 1 0 Mshowa
.02381 .61391 m
.03006 .61391 L
s
[(0.16)] .01131 .61391 1 0 Mshowa
.125 Mabswid
.02381 .07208 m
.02756 .07208 L
s
.02381 .09214 m
.02756 .09214 L
s
.02381 .11221 m
.02756 .11221 L
s
.02381 .15235 m
.02756 .15235 L
s
.02381 .17242 m
.02756 .17242 L
s
.02381 .19248 m
.02756 .19248 L
s
.02381 .23262 m
.02756 .23262 L
s
.02381 .25269 m
.02756 .25269 L
s
.02381 .27276 m
.02756 .27276 L
s
.02381 .31289 m
.02756 .31289 L
s
.02381 .33296 m
.02756 .33296 L
s
.02381 .35303 m
.02756 .35303 L
s
.02381 .39316 m
.02756 .39316 L
s
.02381 .41323 m
.02756 .41323 L
s
.02381 .4333 m
.02756 .4333 L
s
.02381 .47344 m
.02756 .47344 L
s
.02381 .4935 m
.02756 .4935 L
s
.02381 .51357 m
.02756 .51357 L
s
.02381 .55371 m
.02756 .55371 L
s
.02381 .57378 m
.02756 .57378 L
s
.02381 .59384 m
.02756 .59384 L
s
.02381 .03194 m
.02756 .03194 L
s
.02381 .01187 m
.02756 .01187 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.5 Mabswid
.02381 .15355 m
.02598 .08259 L
.02731 .13656 L
.02928 .10843 L
.03091 .11799 L
.03266 .11107 L
.03432 .11165 L
.03599 .1093 L
.03764 .10914 L
.03928 .10662 L
.04089 .10513 L
.04248 .10394 L
.04406 .10407 L
.04564 .10578 L
.04724 .11004 L
.0489 .11861 L
.05065 .13867 L
.05265 .1624 L
.05492 .13344 L
.05686 .14447 L
.05892 .14407 L
.06098 .15479 L
.06316 .14508 L
.06524 .14796 L
.06734 .14559 L
.06942 .14736 L
.07152 .14813 L
.07362 .15053 L
.07576 .15311 L
.07793 .1566 L
.08013 .16061 L
.08239 .16537 L
.0847 .17081 L
.08708 .17705 L
.08954 .18413 L
.09207 .19216 L
.09471 .20125 L
.09745 .21153 L
.10031 .2231 L
.10332 .23602 L
.10647 .2502 L
.1098 .26529 L
.11331 .28048 L
.117 .29456 L
.12085 .29506 L
.12472 .28945 L
.12851 .28346 L
.13224 .27674 L
.13588 .27073 L
.13945 .26542 L
Mistroke
.14296 .26117 L
.14642 .258 L
.14984 .25596 L
.15323 .25503 L
.15662 .25522 L
.16 .25649 L
.16341 .25883 L
.16684 .26224 L
.17031 .26672 L
.17383 .2723 L
.17742 .27899 L
.18109 .28687 L
.18486 .29599 L
.18873 .30647 L
.19273 .31844 L
.19687 .33207 L
.20118 .3476 L
.20567 .36539 L
.21037 .38595 L
.21532 .41013 L
.22056 .43951 L
.22615 .47766 L
.23219 .53578 L
.23893 .56653 L
.24603 .58241 L
.25333 .5703 L
.26048 .56338 L
.26755 .55607 L
.27453 .55199 L
.28146 .55214 L
.2884 .56173 L
.29545 .59258 L
.30286 .58723 L
.31022 .5406 L
.31701 .48606 L
.32316 .36337 L
.32784 .30041 L
.33176 .27736 L
.33541 .23987 L
.33862 .21341 L
.3415 .18687 L
.34407 .16503 L
.34638 .14549 L
.34846 .12885 L
.35034 .11442 L
.35204 .10205 L
.3536 .0914 L
.35503 .08228 L
.35635 .07448 L
.35758 .06784 L
Mistroke
.35872 .06223 L
.35981 .05756 L
.36083 .05376 L
.36181 .0508 L
.36276 .04872 L
.36367 .04761 L
.36458 .04771 L
.36549 .0497 L
.36642 .05614 L
.36743 .05481 L
.36842 .04495 L
.3693 .04197 L
.37013 .03793 L
.37093 .03619 L
.3717 .03609 L
.37246 .04296 L
.37332 .03529 L
.37407 .02758 L
.37474 .02455 L
.37537 .0212 L
.37596 .01918 L
.37653 .01744 L
.37708 .0163 L
.37761 .01548 L
.37813 .01502 L
.37865 .01482 L
.37916 .01487 L
.37968 .01514 L
.3802 .0156 L
.38072 .01625 L
.38125 .01707 L
.38179 .01806 L
.38235 .01921 L
.38292 .02053 L
.3835 .02201 L
.3841 .02366 L
.38472 .02548 L
.38536 .02748 L
.38603 .02967 L
.38672 .03206 L
.38744 .03464 L
.38819 .03744 L
.38898 .04046 L
.3898 .04372 L
.39066 .04722 L
.39156 .05097 L
.39251 .05501 L
.3935 .05934 L
.39455 .06401 L
.39565 .06908 L
Mistroke
.39681 .07462 L
.39804 .08077 L
.39934 .08774 L
.40073 .09591 L
.40221 .10411 L
.4038 .11277 L
.40548 .12309 L
.40729 .1357 L
.40925 .15316 L
.41142 .14449 L
.41348 .15208 L
.41563 .15175 L
.41778 .15626 L
.41999 .15896 L
.42223 .16349 L
.42452 .16807 L
.42686 .17378 L
.42928 .18019 L
.43177 .1877 L
.43435 .19627 L
.43703 .2061 L
.43983 .21725 L
.44276 .22977 L
.44584 .24352 L
.44909 .25806 L
.45251 .27244 L
.4561 .28084 L
.4598 .27773 L
.46345 .27442 L
.46707 .26843 L
.47061 .26288 L
.47409 .25733 L
.4775 .25263 L
.48086 .24871 L
.48417 .24579 L
.48744 .24384 L
.49069 .24289 L
.49393 .24292 L
.49717 .24393 L
.50043 .24589 L
.5037 .24879 L
.50701 .25265 L
.51037 .25747 L
.51378 .26326 L
.51726 .27007 L
.52083 .27794 L
.52449 .28695 L
.52825 .29717 L
.53214 .30871 L
.53617 .32172 L
Mistroke
.54035 .33638 L
.5447 .35295 L
.54926 .37178 L
.55404 .39338 L
.55907 .41866 L
.56441 .44933 L
.57012 .48943 L
.5763 .55265 L
.58324 .5726 L
.59042 .59213 L
.59783 .57002 L
.60498 .56701 L
.61209 .55662 L
.61908 .55373 L
.62603 .55386 L
.63299 .56596 L
.64009 .60332 L
.64763 .57582 L
.65485 .53261 L
.66155 .4728 L
.66754 .34428 L
.67199 .28841 L
.67577 .26554 L
.67928 .23093 L
.68238 .20569 L
.68517 .18069 L
.68767 .15988 L
.68992 .14128 L
.69194 .12537 L
.69378 .11154 L
.69545 .09965 L
.69698 .0894 L
.69838 .08061 L
.69968 .07307 L
.70089 .06665 L
.70203 .06123 L
.7031 .05672 L
.70411 .05307 L
.70508 .05025 L
.70602 .0483 L
.70694 .04732 L
.70784 .04761 L
.70875 .04991 L
.70968 .05737 L
.7107 .0526 L
.71167 .04461 L
.71254 .04119 L
.71337 .03755 L
.71416 .03589 L
.71492 .03623 L
Mistroke
.71569 .04509 L
.71657 .03265 L
.7173 .02767 L
.71797 .02374 L
.71859 .02095 L
.71917 .01878 L
.71974 .01721 L
.72028 .01607 L
.72081 .01532 L
.72133 .01488 L
.72185 .01472 L
.72236 .01479 L
.72287 .01508 L
.72339 .01557 L
.72392 .01624 L
.72445 .01708 L
.72499 .01808 L
.72554 .01925 L
.72611 .02059 L
.72669 .02208 L
.72729 .02375 L
.72792 .02559 L
.72856 .02761 L
.72923 .02981 L
.72992 .03221 L
.73064 .03482 L
.7314 .03763 L
.73218 .04067 L
.73301 .04394 L
.73387 .04745 L
.73478 .05123 L
.73572 .05528 L
.73672 .05962 L
.73777 .06431 L
.73888 .0694 L
.74005 .07496 L
.74128 .08115 L
.74259 .08816 L
.74398 .0964 L
.74547 .10453 L
.74705 .1133 L
.74874 .12368 L
.75056 .13646 L
.75253 .15279 L
.75469 .14476 L
.75676 .15207 L
.75891 .15192 L
.76106 .15637 L
.76327 .15915 L
.76551 .16368 L
Mistroke
.7678 .16832 L
.77015 .17405 L
.77257 .18052 L
.77506 .18807 L
.77765 .1967 L
.78033 .20659 L
.78314 .21781 L
.78608 .23039 L
.78917 .2442 L
.79243 .25875 L
.79585 .27309 L
.79945 .28072 L
.80315 .27763 L
.8068 .27413 L
.81041 .26817 L
.81395 .26259 L
.81743 .25709 L
.82084 .25242 L
.82419 .24855 L
.8275 .24567 L
.83077 .24377 L
.83402 .24286 L
.83726 .24294 L
.8405 .24399 L
.84375 .246 L
.84703 .24895 L
.85034 .25285 L
.8537 .25771 L
.85712 .26355 L
.8606 .27041 L
.86417 .27834 L
.86783 .28739 L
.87161 .29767 L
.8755 .30928 L
.87953 .32236 L
.88372 .33711 L
.88809 .35377 L
.89265 .37272 L
.89744 .39447 L
.90249 .41994 L
.90785 .45093 L
.91357 .49166 L
.91978 .55684 L
.92677 .57227 L
.93395 .59379 L
.94138 .56872 L
.94851 .5673 L
.95563 .55617 L
.96261 .55386 L
.96956 .52647 L
Mistroke
.97619 .52647 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[71]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`006`3oool0
0`000000oooo0?ooo`3o0?ooo`<0oooo000K0?ooo`030000003oool0oooo0?l0oooo0`3oool001/0
oooo00<000000?ooo`3oool0o`3oool30?ooo`006`3oool2000005`0oooo00<000000?ooo`3oool0
E`3oool2000004/0oooo000K0?ooo`030000003oool0oooo05X0oooo0`00001H0?ooo`800000B`3o
ool001/0oooo00<000000?ooo`3oool0FP3oool00`000000oooo0000001H0?ooo`<00000BP3oool0
01/0oooo00<000000?ooo`3oool0FP3oool00`000000oooo0000001G0?ooo`040000003oool0oooo
000004X0oooo000K0?ooo`030000003oool0oooo05T0oooo00<000000?ooo`3oool00P00001F0?oo
o`040000003oool0oooo000004X0oooo000K0?ooo`800000FP3oool01@000000oooo0?ooo`3oool0
000005H0oooo00D000000?ooo`3oool0oooo000000190?ooo`006`3oool00`000000oooo0?ooo`1I
0?ooo`050000003oool0oooo0?ooo`000000E@3oool2000000<0oooo00<000000?ooo`3oool0A`3o
ool001/0oooo00<000000?ooo`3oool0F03oool2000000<0oooo00<000000?ooo`3oool0D`3oool2
000000<0oooo00<000000?ooo`3oool0A`3oool00080oooo0P0000040?ooo`030000003oool0oooo
00<0oooo0P0000030?ooo`@00000103oool00`000000oooo0?ooo`1H0?ooo`800000103oool00`00
0000oooo0?ooo`1B0?ooo`8000000`3oool00`000000oooo0?ooo`170?ooo`0000D0oooo0000003o
ool0oooo000000080?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`1F0?ooo`<000001@3oool00`000000oooo0?ooo`1@0?ooo`@00000103oool0
0`000000oooo0?ooo`160?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0
oooo000000<0oooo00<000000?ooo`3oool0103oool2000005L0oooo0`0000050?ooo`030000003o
ool0oooo0500oooo0`0000050?ooo`030000003oool0oooo04H0oooo00001@3oool000000?ooo`3o
ool0000000P0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo05H0oooo0`0000050?ooo`030000003oool0oooo0500oooo0`0000050?ooo`030000
003oool0oooo04H0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool0
00001@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo05H0oooo00<000000?ooo`3o
ool01P3oool00`000000oooo0?ooo`1?0?ooo`8000001P3oool00`000000oooo0?ooo`160?ooo`00
00D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool00000103oool00`000000oooo0?ooo`1E0?ooo`030000003oool0oooo00L0oooo00<00000
0?ooo`3oool0CP3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo04H0oooo00020?oo
o`8000002P3oool2000000@0oooo0P0000050?ooo`030000003oool0oooo05D0oooo00<000000?oo
o`3oool01`3oool00`000000oooo0?ooo`1>0?ooo`030000003oool0oooo00L0oooo00<000000?oo
o`3oool0A@3oool001/0oooo00<000000?ooo`3oool0E@3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo04h0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`150?ooo`00
6`3oool2000005H0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`1>0?ooo`030000
003oool0oooo00L0oooo00<000000?ooo`3oool0A@3oool001/0oooo00<000000?ooo`3oool0E@3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo04h0oooo00<000000?ooo`3oool01`3o
ool00`000000oooo0?ooo`150?ooo`006`3oool00`000000oooo0?ooo`1D0?ooo`030000003oool0
oooo00T0oooo00<000000?ooo`3oool0C03oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo04@0oooo000K0?ooo`030000003oool0oooo05@0oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`1<0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0A03oool001/0
oooo00<000000?ooo`3oool0E03oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo04`0
oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`140?ooo`006`3oool2000005D0oooo
00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`1<0?ooo`030000003oool0oooo00T0oooo
00<000000?ooo`3oool0A03oool001/0oooo00<000000?ooo`3oool0E03oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo04X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?oo
o`140?ooo`006`3oool00`000000oooo0?ooo`1D0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool0BP3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo04@0oooo000K0?oo
o`040000003oool0oooo0?ooo`800000D@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0
oooo04X0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`130?ooo`006`3oool20000
00050?ooo`00000000000?ooo`000000D03oool00`000000oooo0?ooo`0:0?ooo`030000003oool0
oooo04X0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`130?ooo`006`3oool40000
0080oooo00<000000?ooo`3oool0C@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo
04X0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`130?ooo`006`3oool3000000<0
oooo00<000000?ooo`3oool0C@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo04X0
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`130?ooo`006`3oool3000000<0oooo
00<000000?ooo`3oool0C@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo04T0oooo
00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`130?ooo`006`3oool2000000@0oooo00<0
00000?ooo`3oool0C@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo04T0oooo00<0
00000?ooo`3oool0303oool00`000000oooo0?ooo`120?ooo`000P3oool2000000@0oooo00<00000
0?ooo`3oool00`3oool2000000D0oooo00<000000?ooo`3oool00`3oool2000000@0oooo00<00000
0?ooo`3oool0C@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo04T0oooo00<00000
0?ooo`3oool0303oool00`000000oooo0?ooo`120?ooo`0000D0oooo0000003oool0oooo00000008
0?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool00`3oool2000000D0oooo00<0
00000?ooo`3oool0C03oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo04T0oooo00<0
00000?ooo`3oool0303oool00`000000oooo0?ooo`120?ooo`0000D0oooo0000003oool0oooo0000
00080?ooo`050000003oool0oooo0000003oool01@0000040?ooo`8000001@3oool2000004d0oooo
00<000000?ooo`3oool0303oool00`000000oooo0?ooo`190?ooo`030000003oool0oooo00`0oooo
00<000000?ooo`3oool0@P3oool000050?ooo`000000oooo0?ooo`000000203oool01P000000oooo
0?ooo`000000oooo00000080oooo00<000000?ooo`3oool00`3oool2000000D0oooo0P00001=0?oo
o`030000003oool0oooo00`0oooo00<000000?ooo`3oool0B03oool00`000000oooo0?ooo`0=0?oo
o`030000003oool0oooo0480oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?oo
o`3oool000000P3oool00`000000oooo000000050?ooo`030000003oool0oooo00@0oooo0`00001;
0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool0B03oool00`000000oooo0?ooo`0=
0?ooo`030000003oool0oooo0480oooo00001@3oool000000?ooo`3oool0000000P0oooo00@00000
0?ooo`3oool000000`3oool2000000D0oooo00<000000?ooo`3oool0103oool5000004T0oooo00<0
00000?ooo`3oool03P3oool00`000000oooo0?ooo`170?ooo`030000003oool0oooo00d0oooo0P00
00130?ooo`000P3oool2000000X0oooo0P0000050?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool0103oool7000004L0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`17
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0@@3oool001/0oooo0P0000050?oo
o`040000003oool0oooo000000<0oooo00<000000?ooo`3oool0A03oool00`000000oooo0?ooo`0>
0?ooo`800000A`3oool00`000000oooo0?ooo`0?0?ooo`800000@P3oool001/0oooo00<000000?oo
o`3oool0103oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo04@0oooo00<00000
0?ooo`3oool0403oool00`000000oooo0?ooo`140?ooo`030000003oool0oooo0140oooo00<00000
0?ooo`3oool0?`3oool001/0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?oo
o`030000003oool0oooo04<0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`130?oo
o`030000003oool0oooo0140oooo00<000000?ooo`3oool0?`3oool001/0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo04<0oooo00<000000?ooo`3o
ool04@3oool00`000000oooo0?ooo`130?ooo`030000003oool0oooo0180oooo00<000000?ooo`3o
ool0?P3oool001/0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`130?ooo`030000
003oool0oooo0180oooo00<000000?ooo`3oool0@P3oool00`000000oooo0?ooo`0C0?ooo`030000
003oool0oooo03d0oooo000K0?ooo`8000003P3oool00`000000oooo0?ooo`120?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool0@P3oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo03d0oooo000K0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0@@3oool0
0`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0480oooo00<000000?ooo`3oool04`3oool0
0`000000oooo0?ooo`0m0?ooo`006`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo
0400oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`110?ooo`030000003oool0oooo
01@0oooo00<000000?ooo`3oool0?03oool001/0oooo00<000000?ooo`3oool03P3oool00`000000
oooo0?ooo`100?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0@@3oool00`000000
oooo0?ooo`0D0?ooo`030000003oool0oooo03`0oooo000K0?ooo`030000003oool0oooo00h0oooo
00<000000?ooo`3oool0@03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0400oooo
00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0k0?ooo`006`3oool200000100oooo00<0
00000?ooo`3oool0?`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0400oooo00<0
00000?ooo`3oool05@3oool00`000000oooo0?ooo`0k0?ooo`006`3oool00`000000oooo0?ooo`0?
0?ooo`030000003oool0oooo03l0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0o
0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0>`3oool001/0oooo00<000000?oo
o`3oool0403oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool0?`3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo03/0oooo000K0?ooo`03
0000003oool0oooo0100oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0G0?ooo`03
0000003oool0oooo03h0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0k0?ooo`00
0P3oool2000000@0oooo00<000000?ooo`3oool00`3oool2000000@0oooo0P0000050?ooo`030000
003oool0oooo0100oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0G0?ooo`030000
003oool0oooo03h0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`0j0?ooo`0000D0
oooo0000003oool0oooo000000080?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3o
ool00000103oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo03d0oooo00<000000?oo
o`3oool05`3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo01L0oooo00<000000?oo
o`3oool0>P3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo000000@0oooo0P00000A0?ooo`030000003oool0oooo03d0oooo00<0
00000?ooo`3oool0603oool00`000000oooo0?ooo`0m0?ooo`030000003oool0oooo01P0oooo00<0
00000?ooo`3oool0>@3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?oo
o`0000020?ooo`<000001@3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo03d0oooo
00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0m0?ooo`030000003oool0oooo01P0oooo
00<000000?ooo`3oool0>@3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo
0?ooo`0000020?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04@3oool00`000000
oooo0?ooo`0l0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0?03oool00`000000
oooo0?ooo`0I0?ooo`030000003oool0oooo03T0oooo00001@3oool000000?ooo`3oool0000000P0
oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
0140oooo00<000000?ooo`3oool0?03oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo
03/0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0i0?ooo`000P3oool2000000X0
oooo0P0000040?ooo`<00000103oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo03/0
oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`0k0?ooo`030000003oool0oooo01T0
oooo00<000000?ooo`3oool0>@3oool001/0oooo0P00000C0?ooo`030000003oool0oooo03X0oooo
00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`0k0?ooo`030000003oool0oooo01T0oooo
00<000000?ooo`3oool0>@3oool001/0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?oo
o`0j0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0>`3oool00`000000oooo0?oo
o`0I0?ooo`030000003oool0oooo03T0oooo000K0?ooo`030000003oool0oooo0180oooo00<00000
0?ooo`3oool0>P3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo03/0oooo00<00000
0?ooo`3oool06P3oool00`000000oooo0?ooo`0h0?ooo`006`3oool00`000000oooo0?ooo`0B0?oo
o`030000003oool0oooo03X0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`080?oo
o`@00000;`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo00P0oooo1000000/0?oo
o`006`3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo03T0oooo00<000000?ooo`3o
ool06P3oool00`000000oooo0?ooo`070?ooo`8000000`3oool2000002d0oooo00<000000?ooo`3o
ool06P3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0080oooo0P00000Z0?ooo`00
6`3oool2000001@0oooo00<000000?ooo`3oool0>@3oool00`000000oooo0?ooo`0J0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0Y0?ooo`030000
003oool0oooo01/0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo02P0oooo000K0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool0203o
ool4000002d0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`050?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0L0?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0X0?ooo`006`3oool0
0`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo02P0oooo00<000000?ooo`3oool0703oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool09`3oool0
01/0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool0:03oool00`000000oooo0?ooo`0L0?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0X0?ooo`030000003oool0oooo
01`0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo
02H0oooo000K0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`060?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool0703oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0:03oool00`000000
oooo0?ooo`0L0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0203oool00`000000
oooo0?ooo`0V0?ooo`006`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo01d0oooo
00D000000?ooo`3oool0oooo0000000<0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3o
ool07@3oool01@000000oooo0?ooo`3oool0000000`0oooo00<000000?ooo`3oool09@3oool001/0
oooo0P00000E0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool01`3oool00`000000
oooo0?ooo`0W0?ooo`030000003oool0oooo01d0oooo00@000000?ooo`3oool000003@3oool00`00
0000oooo0?ooo`0W0?ooo`030000003oool0oooo01d0oooo00@000000?ooo`3oool000003@3oool0
0`000000oooo0?ooo`0U0?ooo`006`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo
0080oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0V0?ooo`030000003oool0oooo
01d0oooo00<000000?ooo`0000003`3oool00`000000oooo0?ooo`0V0?ooo`030000003oool0oooo
01d0oooo00<000000?ooo`0000003`3oool00`000000oooo0?ooo`0T0?ooo`006`3oool00`000000
oooo0?ooo`0D0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool02@3oool00`000000
oooo0?ooo`0V0?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool03P3oool00`000000
oooo0?ooo`0U0?ooo`030000003oool0oooo01l0oooo00<000000?ooo`3oool03P3oool00`000000
oooo0?ooo`0T0?ooo`000P3oool2000000@0oooo00<000000?ooo`3oool00`3oool2000000@0oooo
0P0000050?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool00P3oool00`000000oooo
0?ooo`0:0?ooo`030000003oool0oooo02D0oooo00<000000?ooo`3oool0<03oool00`000000oooo
0?ooo`0T0?ooo`030000003oool0oooo0340oooo00<000000?ooo`3oool08`3oool000050?ooo`00
0000oooo0?ooo`000000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
00@0oooo00<000000?ooo`3oool0503oool01@000000oooo0?ooo`3oool0000000d0oooo00<00000
0?ooo`3oool09@3oool00`000000oooo0?ooo`0`0?ooo`030000003oool0oooo02@0oooo00<00000
0?ooo`3oool0<@3oool00`000000oooo0?ooo`0S0?ooo`0000D0oooo0000003oool0oooo00000008
0?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool2000001H0oooo
00<000000?ooo`0000003`3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo0380oooo
00<000000?ooo`3oool08`3oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo02<0oooo
00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool000000`3oool2000000D0
oooo00<000000?ooo`3oool05@3oool200000100oooo00<000000?ooo`3oool08`3oool00`000000
oooo0?ooo`0b0?ooo`030000003oool0oooo02<0oooo00<000000?ooo`3oool0<P3oool00`000000
oooo0?ooo`0R0?ooo`0000D0oooo0000003oool0oooo000000080?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`0W0?ooo`030000003oool0
oooo02<0oooo00<000000?ooo`3oool0<P3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0
oooo0380oooo00<000000?ooo`3oool08P3oool000050?ooo`000000oooo0?ooo`000000203oool0
10000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool0
:03oool00`000000oooo0?ooo`0R0?ooo`030000003oool0oooo03<0oooo00<000000?ooo`3oool0
8P3oool00`000000oooo0?ooo`0c0?ooo`030000003oool0oooo0240oooo00020?ooo`8000002P3o
ool2000000@0oooo0P0000050?ooo`030000003oool0oooo02P0oooo00<000000?ooo`3oool08P3o
ool00`000000oooo0?ooo`0c0?ooo`030000003oool0oooo0280oooo00<000000?ooo`3oool0<`3o
ool00`000000oooo0?ooo`0Q0?ooo`006`3oool2000002T0oooo00<000000?ooo`3oool08P3oool0
0`000000oooo0?ooo`0c0?ooo`030000003oool0oooo0280oooo00<000000?ooo`3oool0<`3oool0
0`000000oooo0?ooo`0Q0?ooo`006`3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo
0240oooo00<000000?ooo`3oool0<`3oool00`000000oooo0?ooo`0R0?ooo`030000003oool0oooo
03<0oooo00<000000?ooo`3oool08@3oool001/0oooo00<000000?ooo`3oool0:@3oool00`000000
oooo0?ooo`0Q0?ooo`030000003oool0oooo03@0oooo00<000000?ooo`3oool0803oool00`000000
oooo0?ooo`0e0?ooo`030000003oool0oooo0200oooo000K0?ooo`030000003oool0oooo02X0oooo
00<000000?ooo`3oool0803oool00`000000oooo0?ooo`0d0?ooo`030000003oool0oooo0200oooo
00<000000?ooo`3oool0=@3oool00`000000oooo0?ooo`0P0?ooo`006`3oool00`000000oooo0?oo
o`0Z0?ooo`030000003oool0oooo0200oooo00<000000?ooo`3oool0=03oool00`000000oooo0?oo
o`0P0?ooo`030000003oool0oooo03D0oooo00<000000?ooo`3oool0803oool001/0oooo00<00000
0?ooo`3oool0:`3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo03@0oooo00<00000
0?ooo`3oool0803oool00`000000oooo0?ooo`0e0?ooo`030000003oool0oooo0200oooo000K0?oo
o`800000;03oool00`000000oooo0?ooo`0<0?ooo`<000003`3oool00`000000oooo0?ooo`0X0?oo
o`@000000`3oool2000000D0oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`0<0?oo
o`@000000P3oool300000240oooo00<000000?ooo`3oool04`3oool4000000<0oooo0P0000030?oo
o`006`3oool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3o
ool0303oool00`000000oooo0?ooo`0Z0?ooo`050000003oool0oooo0?ooo`0000000P3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo01l0oooo00<000000?ooo`3oool03P3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool04`3oool00`00
0000oooo0?ooo`030?ooo`040000003oool0oooo00000080oooo000K0?ooo`030000003oool0oooo
02`0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo
02X0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo0080oooo00<00000
0?ooo`3oool07`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00@0oooo00<00000
0?ooo`3oool07P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo0080oooo00@00000
0?ooo`3oool000000P3oool001/0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0;
0?ooo`<000003`3oool00`000000oooo0?ooo`0Z0?ooo`050000003oool0oooo0?ooo`0000000P3o
ool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool03P3o
ool010000000oooo0?ooo`3oool300000240oooo00<000000?ooo`3oool05@3oool01@000000oooo
0?ooo`3oool000000080oooo0@0000010?ooo`40oooo000K0?ooo`030000003oool0oooo02`0oooo
00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo02X0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool07P3oool00`000000oooo0?ooo`0>0?ooo`050000003oool0oooo0?ooo`000000903oool00`00
0000oooo0?ooo`0E0?ooo`040000003oool0oooo00000080oooo0@0000010?ooo`40oooo000K0?oo
o`800000;@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00l0oooo00<000000?oo
o`3oool0:03oool3000000<0oooo00@000000?ooo`3oool000001P3oool00`000000oooo0?ooo`0M
0?ooo`030000003oool0oooo00`0oooo0`0000030?ooo`030000003oool0oooo0280oooo00<00000
0?ooo`3oool04P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo
000K0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool02`3oool4000000h0oooo00<0
00000?ooo`3oool0:P3oool00`000000oooo0?ooo`020?ooo`8000001`3oool00`000000oooo0?oo
o`0M0?ooo`030000003oool0oooo00h0oooo00@000000?ooo`3oool0oooo1000000Q0?ooo`030000
003oool0oooo01<0oooo0P0000040?ooo`8000000`3oool001/0oooo00<000000?ooo`3oool0;@3o
ool00`000000oooo0?ooo`0L0?ooo`030000003oool0oooo03P0oooo00<000000?ooo`3oool07@3o
ool00`000000oooo0?ooo`0h0?ooo`030000003oool0oooo01d0oooo000K0?ooo`030000003oool0
oooo02d0oooo00<000000?ooo`3oool0703oool00`000000oooo0?ooo`0i0?ooo`030000003oool0
oooo01`0oooo00<000000?ooo`3oool0>03oool00`000000oooo0?ooo`0M0?ooo`006`3oool00`00
0000oooo0?ooo`0]0?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool0>@3oool00`00
0000oooo0?ooo`0L0?ooo`030000003oool0oooo03T0oooo00<000000?ooo`3oool0703oool001D0
ooooo`00000:00000080oooo000K0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
2P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00L0oooo00@000000?ooo`3oool0
00002`3oool00`000000oooo0?ooo`0:0?ooo`040000003oool0oooo000000T0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool02@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`0000002P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00@0oooo000K0?ooo`03
0000003oool0oooo02h0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo02d0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0L0?ooo`03
0000003oool0oooo0140oooo00<000000?ooo`3oool09@3oool00`000000oooo0?ooo`0E0?ooo`03
0000003oool0oooo00@0oooo000K0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3oool0
6`3oool00`000000oooo0?ooo`0i0?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool0
>@3oool00`000000oooo0?ooo`0L0?ooo`006`3oool00`000000oooo0?ooo`0^0?ooo`030000003o
ool0oooo01/0oooo00<000000?ooo`3oool0>P3oool00`000000oooo0?ooo`0K0?ooo`030000003o
ool0oooo03X0oooo00<000000?ooo`3oool06`3oool001/0oooo00<000000?ooo`3oool0;P3oool0
0`000000oooo0?ooo`0K0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool06`3oool0
0`000000oooo0?ooo`0j0?ooo`030000003oool0oooo01/0oooo000K0?ooo`800000;`3oool00`00
0000oooo0?ooo`0K0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool06`3oool00`00
0000oooo0?ooo`0j0?ooo`030000003oool0oooo01/0oooo000K0?ooo`030000003oool0oooo02h0
oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`0k0?ooo`030000003oool0oooo01/0
oooo00<000000?ooo`3oool0>P3oool00`000000oooo0?ooo`0K0?ooo`006`3oool00`000000oooo
0?ooo`0_0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0>`3oool00`000000oooo
0?ooo`0K0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool06`3oool001/0oooo00<0
00000?ooo`3oool0;`3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo03/0oooo00<0
00000?ooo`3oool06`3oool00`000000oooo0?ooo`0j0?ooo`030000003oool0oooo01/0oooo000K
0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0l
0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0>`3oool00`000000oooo0?ooo`0J
0?ooo`006`3oool200000300oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0l0?oo
o`030000003oool0oooo01T0oooo00<000000?ooo`3oool0?03oool00`000000oooo0?ooo`0J0?oo
o`006`3oool00`000000oooo0?ooo`0_0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3o
ool0?03oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo03`0oooo00<000000?ooo`3o
ool06P3oool001/0oooo00<000000?ooo`3oool0<03oool00`000000oooo0?ooo`0H0?ooo`030000
003oool0oooo03`0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0l0?ooo`030000
003oool0oooo01X0oooo000K0?ooo`030000003oool0oooo0300oooo00<000000?ooo`3oool0603o
ool00`000000oooo0?ooo`0l0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0?03o
ool00`000000oooo0?ooo`0J0?ooo`006`3oool00`000000oooo0?ooo`0`0?ooo`030000003oool0
oooo01P0oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0
oooo03`0oooo00<000000?ooo`3oool06P3oool001/0oooo0P00000a0?ooo`030000003oool0oooo
01P0oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo
03d0oooo00<000000?ooo`3oool06@3oool001/0oooo00<000000?ooo`3oool0<@3oool00`000000
oooo0?ooo`0G0?ooo`030000003oool0oooo03d0oooo00<000000?ooo`3oool0603oool00`000000
oooo0?ooo`0m0?ooo`030000003oool0oooo01T0oooo000K0?ooo`030000003oool0oooo0340oooo
00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`0m0?ooo`030000003oool0oooo01P0oooo
00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`0I0?ooo`006`3oool00`000000oooo0?oo
o`0a0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0?P3oool00`000000oooo0?oo
o`0G0?ooo`030000003oool0oooo03d0oooo00<000000?ooo`3oool06@3oool00080oooo0P000004
0?ooo`030000003oool0oooo0080oooo100000020?ooo`@00000103oool00`000000oooo0?ooo`0a
0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0?P3oool00`000000oooo0?ooo`0G
0?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool0603oool000050?ooo`000000oooo
0?ooo`0000002P3oool01@000000oooo0?ooo`3oool0000000L0oooo00<000000?ooo`3oool0<@3o
ool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool05`3o
ool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo01P0oooo00001@3oool000000?ooo`3o
ool0000000X0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`040?ooo`800000<P3o
ool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool05`3o
ool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo01P0oooo00001@3oool000000?ooo`3o
ool0000000X0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo0340oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0o0?ooo`030000003o
ool0oooo01L0oooo00<000000?ooo`3oool0?P3oool00`000000oooo0?ooo`0H0?ooo`0000D0oooo
0000003oool0oooo0000000:0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`0b0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0?`3o
ool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool0603o
ool000050?ooo`000000oooo0?ooo`000000203oool3000000<0oooo00@000000?ooo`3oool00000
103oool00`000000oooo0?ooo`0b0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0
?`3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool0
603oool00080oooo0P00000;0?ooo`030000003oool0oooo0080oooo0P0000050?ooo`030000003o
ool0oooo0380oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`100?ooo`030000003o
ool0oooo01H0oooo00<000000?ooo`3oool0?P3oool00`000000oooo0?ooo`0H0?ooo`006`3oool2
000003<0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`100?ooo`030000003oool0
oooo01H0oooo00<000000?ooo`3oool0?`3oool00`000000oooo0?ooo`0G0?ooo`006`3oool00`00
0000oooo0?ooo`0b0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0@03oool00`00
0000oooo0?ooo`0F0?ooo`030000003oool0oooo03l0oooo00<000000?ooo`3oool05`3oool001/0
oooo00<000000?ooo`3oool0<P3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0400
oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0o0?ooo`030000003oool0oooo01L0
oooo000K0?ooo`030000003oool0oooo0380oooo00<000000?ooo`3oool05@3oool00`000000oooo
0?ooo`100?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0?`3oool00`000000oooo
0?ooo`0G0?ooo`006`3oool00`000000oooo0?ooo`0b0?ooo`030000003oool0oooo01D0oooo00<0
00000?ooo`3oool0@03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0400oooo00<0
00000?ooo`3oool05`3oool001/0oooo0P00000d0?ooo`030000003oool0oooo01@0oooo00<00000
0?ooo`3oool0@03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0400oooo00<00000
0?ooo`3oool05`3oool001/0oooo00<000000?ooo`3oool0<`3oool00`000000oooo0?ooo`0D0?oo
o`030000003oool0oooo0400oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`100?oo
o`030000003oool0oooo01L0oooo000K0?ooo`030000003oool0oooo03<0oooo00<000000?ooo`3o
ool0503oool00`000000oooo0?ooo`100?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3o
ool0@03oool00`000000oooo0?ooo`0G0?ooo`006`3oool00`000000oooo0?ooo`0c0?ooo`030000
003oool0oooo01@0oooo00<000000?ooo`3oool0@03oool00`000000oooo0?ooo`0E0?ooo`030000
003oool0oooo0400oooo00<000000?ooo`3oool05`3oool001/0oooo00<000000?ooo`3oool0<`3o
ool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo0440oooo00<000000?ooo`3oool0503o
ool00`000000oooo0?ooo`100?ooo`030000003oool0oooo01L0oooo000K0?ooo`800000=03oool0
0`000000oooo0?ooo`0D0?ooo`030000003oool0oooo0440oooo00<000000?ooo`3oool0503oool0
0`000000oooo0?ooo`110?ooo`030000003oool0oooo01H0oooo000K0?ooo`030000003oool0oooo
03<0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`120?ooo`030000003oool0oooo
01@0oooo00<000000?ooo`3oool0@@3oool00`000000oooo0?ooo`0F0?ooo`006`3oool00`000000
oooo0?ooo`0c0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool0@P3oool00`000000
oooo0?ooo`0C0?ooo`030000003oool0oooo0480oooo00<000000?ooo`3oool05P3oool001/0oooo
00<000000?ooo`3oool0=03oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0480oooo
00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`120?ooo`030000003oool0oooo01H0oooo
00020?ooo`800000103oool00`000000oooo0?ooo`020?ooo`@00000103oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo03@0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?oo
o`120?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool0@P3oool00`000000oooo0?oo
o`0;0?ooo`<00000203oool000050?ooo`000000oooo0?ooo`0000002P3oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0=03oool00`000000oooo0?oo
o`0B0?ooo`030000003oool0oooo0480oooo00<000000?ooo`3oool04`3oool00`000000oooo0?oo
o`120?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0203oool000050?ooo`000000
oooo0?ooo`0000002P3oool00`000000oooo0?ooo`05000000@0oooo0P00000e0?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool0@P3oool00`000000oooo0?ooo`0C0?ooo`030000003o
ool0oooo0480oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`090?ooo`0000D0oooo
0000003oool0oooo0000000:0?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0d0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0
@`3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0480oooo00<000000?ooo`3oool0
2P3oool00`000000oooo0?ooo`090?ooo`0000D0oooo0000003oool0oooo0000000:0?ooo`070000
003oool0oooo0?ooo`000000oooo000000050?ooo`030000003oool0oooo03@0oooo00<000000?oo
o`3oool04P3oool00`000000oooo0?ooo`130?ooo`030000003oool0oooo0180oooo00<000000?oo
o`3oool0@`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo00001@3oool0
00000?ooo`3oool0000000P0oooo0`0000040?ooo`8000001@3oool00`000000oooo0?ooo`0e0?oo
o`030000003oool0oooo0140oooo00<000000?ooo`3oool0@`3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo04@0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`090?oo
o`000P3oool2000000/0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo03D0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`130?ooo`03
0000003oool0oooo0140oooo00<000000?ooo`3oool0A03oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo00X0oooo000K0?ooo`800000=P3oool00`000000oooo0?ooo`060?ooo`@00000
1`3oool00`000000oooo0?ooo`130?ooo`030000003oool0oooo00L0oooo100000060?ooo`030000
003oool0oooo04@0oooo00<000000?ooo`3oool01`3oool2000000`0oooo000K0?ooo`030000003o
ool0oooo03D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool0A03oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo0080oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`140?ooo`030000003o
ool0oooo00H0oooo00<000000?ooo`3oool0303oool001/0oooo00<000000?ooo`3oool0=P3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`150?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo04@0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`0=0?ooo`006`3oool00`000000oooo0?ooo`0f0?ooo`030000003oool0oooo
0080oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo
04D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
0080oooo00<000000?ooo`3oool0A@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
00d0oooo000K0?ooo`030000003oool0oooo03H0oooo00D000000?ooo`3oool0oooo000000080?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool0AP3oool010000000oooo0?ooo`3oool3
000000H0oooo00D000000?ooo`3oool0oooo000000180?ooo`030000003oool0oooo0080oooo0P00
000@0?ooo`006`3oool2000003P0oooo00@000000?ooo`3oool000002@3oool01@000000oooo0?oo
o`3oool0000004P0oooo00D000000?ooo`3oool0oooo000000080?ooo`050000003oool0oooo0?oo
o`000000B@3oool01@000000oooo0?ooo`3oool000000140oooo000K0?ooo`030000003oool0oooo
03L0oooo00<000000?ooo`0000002P3oool01@000000oooo0?ooo`3oool0000004P0oooo00@00000
0?ooo`3oool000002P3oool010000000oooo0?ooo`0000190?ooo`040000003oool0oooo00000180
oooo000K0?ooo`030000003oool0oooo03P0oooo0P00000:0?ooo`040000003oool0oooo000004X0
oooo00<000000?ooo`0000002P3oool010000000oooo0?ooo`00001:0?ooo`030000003oool00000
0180oooo000K0?ooo`030000003oool0oooo03P0oooo00<000000?ooo`3oool02@3oool010000000
oooo0?ooo`00001:0?ooo`030000003oool0000000X0oooo00<000000?ooo`000000B`3oool00`00
0000oooo0000000B0?ooo`006`3oool00`000000oooo0?ooo`150?ooo`<00000B`3oool00`000000
oooo0?ooo`090?ooo`030000003oool0000004/0oooo00<000000?ooo`0000004P3oool001/0oooo
00<000000?ooo`3oool0A@3oool00`000000oooo0?ooo`1;0?ooo`030000003oool0oooo00T0oooo
00<000000?ooo`000000C03oool00`000000oooo0?ooo`0A0?ooo`006`3oool200000:40oooo0P00
001<0?ooo`030000003oool0oooo0140oooo000K0?ooo`030000003oool0oooo0:00oooo00<00000
0?ooo`3oool0G`3oool001/0oooo00<000000?ooo`3oool0X03oool00`000000oooo0?ooo`1O0?oo
o`000P3oool2000000@0oooo00<000000?ooo`3oool00P3oool4000000<0oooo0P0000050?ooo`03
0000003oool0oooo0?l0oooo0`3oool000050?ooo`000000oooo0?ooo`0000002P3oool01@000000
oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`3o0?oo
o`<0oooo00001@3oool000000?ooo`3oool0000000X0oooo00D000000?ooo`3oool0oooo00000002
0?ooo`030000003oool0oooo0080oooo0P00003o0?ooo`@0oooo00001@3oool000000?ooo`3oool0
000000X0oooo00@000000?ooo`3oool0oooo0`0000050?ooo`030000003oool0oooo0?l0oooo0`3o
ool000050?ooo`000000oooo0?ooo`0000002P3oool01@000000oooo0?ooo`3oool000000?l0oooo
3@3oool000050?ooo`000000oooo0?ooo`000000203oool3000000<0oooo00<000000?ooo`3oool0
o`3oool;0?ooo`000P3oool2000000/0oooo00<000000?ooo`3oool00P3oool300000?l0oooo2P3o
ool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3o
ool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-2.15285, 0.0023151, \
0.0788363, 0.000941856}}]
}, Open  ]],

Cell[TextData[{
  "For this reason, the default setting for ",
  StyleBox["StepSizeControlParameters", "MR"],
  " is ",
  StyleBox["Automatic", "MR"],
  " , which is interpreted as:"
}], "Text"],

Cell[TextData[{
  "Use the I step controller (",
  
  CounterBox["NumberedEquation", "IController"],
  ") if ",
  StyleBox["StiffnessTest \[Rule] False", "MR"],
  "."
}], "BulletedList"],

Cell[TextData[{
  "Use the PI step controller (",
  
  CounterBox["NumberedEquation", "PIController"],
  ") if ",
  StyleBox["StiffnessTest \[Rule] True", "MR"],
  "."
}], "BulletedList"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Fine-tuning", "Subsection"],

Cell[TextData[{
  "Instead of using (",
  
  CounterBox["NumberedEquation", "PIController"],
  ") directly, it is common practice to use safety factors to ensure that the \
error is acceptable at the next step with high probability, thereby \
preventing unwanted step rejections."
}], "Text"],

Cell[TextData[{
  "The option ",
  StyleBox["StepSizeSafetyFactors\[Rule]{", "MR"],
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`s\_1\)], "MR"], "MR"],
  StyleBox[",", "MR"],
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`s\_2\)], "MR"], "MR"],
  StyleBox["}", "MR"],
  " specifies the safety factors to use in the step size estimate so that (",
  
  CounterBox["NumberedEquation", "PIController"],
  ")",
  " ",
  "becomes:"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`h\_\(n + 1\) = \ 
      h\_n\ s\_1\ \(\((\(\(s\_2\) Tol\)\/\[LeftDoubleBracketingBar]err\_n\
\[RightDoubleBracketingBar])\)\^\(k\_1/\ \(p\& \[Tilde] \)\)\) \(\(\((\
\[LeftDoubleBracketingBar]err\_\(n - 1\)\[RightDoubleBracketingBar]\/\
\[LeftDoubleBracketingBar]err\_n\[RightDoubleBracketingBar])\)\^\(k\_2/\ \
\(p\& \[Tilde] \)\)\)\(\ \)\(.\)\)\)], "NumberedEquation",
  CellTags->"StepSizeSafetyFactors"],

Cell[TextData[{
  "Here ",
  Cell[BoxData[
      \(TraditionalForm\`s\_1\)]],
  " is an absolute factor and ",
  Cell[BoxData[
      \(TraditionalForm\`s\_2\)]],
  " typically scales with the order of the method."
}], "Text"],

Cell[TextData[{
  "The option ",
  StyleBox["StepSizeRatioBounds\[Rule]{", "MR"],
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`sr\_min\)], "MR"], "MR"],
  StyleBox[",", "MR"],
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`sr\_max\)], "MR"], "MR"],
  StyleBox["}", "MR"],
  " specifies bounds on the next step size to take such that:"
}], "Text"],

Cell[BoxData[
    \(TraditionalForm\`sr\_min \[LessEqual] \[LeftBracketingBar]h\_\(n + \
1\)\/h\_n\[RightBracketingBar] \[LessEqual] sr\_max\)], "NumberedEquation",
  CellTags->"StepSizeRatioBounds"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
  CellTags->"s:8"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {
            StyleBox["Coefficients",
              "MR"], 
            StyleBox["EmbeddedExplicitRungeKuttaCoefficients",
              
              "MR"], \(specifies\ the\ coefficients\ of\ the\ explicit\ Runge \
- Kutta\ method\)},
          {
            StyleBox["DifferenceOrder",
              "MR"], 
            StyleBox["Automatic",
              
              "MR"], \(specifies\ the\ order\ of\ local\ \
\(\(accuracy\)\(.\)\)\)},
          {
            StyleBox["EmbeddedDifferenceOrder",
              "MR"], 
            StyleBox["Automatic",
              
              "MR"], \(specifies\ the\ order\ of\ the\ embedded\ method\ in\ \
a\ pair\ of\ explicit\ Runge - Kutta\ methods\)},
          {
            StyleBox["StepSizeControlParameters",
              "MR"], 
            StyleBox["Automatic",
              "MR"], 
            RowBox[{
            "specifies", " ", "the", " ", "PI", " ", "step", " ", "control", 
              " ", "parameters", " ", 
              RowBox[{"(", 
                RowBox[{"see", " ", 
                  RowBox[{"(", 
                    CounterBox["NumberedEquation", "PIController"], ")"}]}], 
                ")"}]}]},
          {
            StyleBox["StepSizeRatioBounds",
              "MR"], \({1\/8, \ 4}\), 
            RowBox[{
            "specifies", " ", "the", " ", "bounds", " ", "on", " ", "a", " ", 
              "relative", " ", "change", " ", "in", " ", "the", " ", "new", 
              " ", "step", " ", "size", " ", 
              RowBox[{"(", 
                RowBox[{"see", " ", 
                  RowBox[{"(", 
                    CounterBox["NumberedEquation", "StepSizeRatioBounds"], 
                    ")"}]}]}]}]},
          {
            StyleBox["StepSizeSafetyFactors",
              "MR"], \({9\/10, 9\/10}\), 
            RowBox[{
            "specifies", " ", "the", " ", "safety", " ", "factors", " ", "to",
               " ", "use", " ", "in", " ", "the", " ", "step", " ", "size", 
              " ", "estimate", " ", 
              RowBox[{"(", 
                RowBox[{"see", " ", 
                  RowBox[{"(", 
                    CounterBox["NumberedEquation", "StepSizeSafetyFactors"], 
                    ")"}]}], ")"}]}]},
          {
            StyleBox["StepSizeStartingError",
              
              "MR"], \(1\/10000\), \(specifies\ the\ initial\ error\ for\ the\
\ numerical\ integration\ when\ using\ PI\ step\ control\)},
          {
            StyleBox["StiffnessTest",
              "MR"], 
            StyleBox["Automatic",
              
              "MR"], \(specifies\ whether\ to\ use\ the\ stiffness\ detection\
\ capability\)}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  "Options of the method ",
  StyleBox["ExplicitRungeKutta", "MR"],
  ". "
}], "Caption"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.1 for Macintosh",
ScreenRectangle->{{33, 1280}, {0, 938}},
AutoGeneratedPackage->None,
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{651, 502},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"ExplicitRungeKutta",
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
Magnification->1,
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "c:1"->{
    Cell[1776, 53, 56, 1, 88, "Chapter",
      CellTags->"c:1"]},
  "s:1"->{
    Cell[1857, 58, 50, 1, 61, "Section",
      CellTags->"s:1"]},
  "EulersMethod"->{
    Cell[2498, 84, 178, 4, 30, "NumberedEquation",
      CellTags->"EulersMethod"]},
  "ExplicitMidpointMethod"->{
    Cell[4214, 147, 473, 11, 69, "NumberedEquation",
      CellTags->"ExplicitMidpointMethod"]},
  "ERKMethod"->{
    Cell[6161, 217, 650, 16, 67, "NumberedEquation",
      CellTags->"ERKMethod"]},
  "RowSumConditions"->{
    Cell[7067, 249, 146, 3, 52, "NumberedEquation",
      CellTags->"RowSumConditions"]},
  "ButcherTable"->{
    Cell[8983, 313, 783, 16, 97, "NumberedEquation",
      CellTags->"ButcherTable"]},
  "ExplicitMidpointButcherTable"->{
    Cell[10350, 355, 338, 10, 70, "NumberedEquation",
      CellTags->"ExplicitMidpointButcherTable"]},
  "FSALConditions"->{
    Cell[11002, 379, 174, 3, 70, "NumberedEquation",
      CellTags->"FSALConditions"]},
  "FSALButcherTable"->{
    Cell[11347, 393, 887, 18, 70, "NumberedEquation",
      CellTags->"FSALButcherTable"]},
  "EmbeddedButcherTable"->{
    Cell[13276, 449, 1038, 20, 137, "NumberedEquation",
      CellTags->"EmbeddedButcherTable"]},
  "HigherOrder"->{
    Cell[14361, 473, 173, 4, 52, "NumberedEquation",
      CellTags->"HigherOrder"]},
  "LowerOrder"->{
    Cell[14537, 479, 188, 5, 52, "NumberedEquation",
      CellTags->"LowerOrder"]},
  "IController"->{
    Cell[17459, 580, 217, 5, 70, "NumberedEquation",
      CellTags->"IController"]},
  "s:2"->{
    Cell[20976, 713, 45, 1, 61, "Section",
      CellTags->"s:2"]},
  "s:3"->{
    Cell[64112, 1632, 55, 1, 70, "Section",
      CellTags->"s:3"]},
  "s:4"->{
    Cell[137133, 3856, 57, 1, 61, "Section",
      CellTags->"s:4"]},
  "s:5"->{
    Cell[148403, 4226, 52, 1, 66, "Section",
      CellTags->"s:5"]},
  "s:6"->{
    Cell[151037, 4315, 47, 1, 66, "Section",
      CellTags->"s:6"]},
  "DahlquistEquation"->{
    Cell[151712, 4343, 223, 4, 30, "NumberedEquation",
      CellTags->"DahlquistEquation"]},
  "StiffnessDevice"->{
    Cell[737733, 47202, 278, 5, 48, "NumberedEquation",
      CellTags->"StiffnessDevice"]},
  "StiffButcherTable"->{
    Cell[739268, 47263, 897, 18, 115, "NumberedEquation",
      CellTags->"StiffButcherTable"]},
  "s:7"->{
    Cell[746606, 47509, 60, 1, 66, "Section",
      CellTags->"s:7"]},
  "PIController"->{
    Cell[806513, 48692, 398, 7, 52, "NumberedEquation",
      CellTags->"PIController"]},
  "StepSizeSafetyFactors"->{
    Cell[919615, 51528, 443, 7, 52, "NumberedEquation",
      CellTags->"StepSizeSafetyFactors"]},
  "StepSizeRatioBounds"->{
    Cell[920646, 51559, 199, 3, 47, "NumberedEquation",
      CellTags->"StepSizeRatioBounds"]},
  "s:8"->{
    Cell[920894, 51568, 52, 1, 61, "Section",
      CellTags->"s:8"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"c:1", 924824, 51679},
  {"s:1", 924901, 51682},
  {"EulersMethod", 924987, 51685},
  {"ExplicitMidpointMethod", 925102, 51688},
  {"ERKMethod", 925216, 51691},
  {"RowSumConditions", 925324, 51694},
  {"ButcherTable", 925434, 51697},
  {"ExplicitMidpointButcherTable", 925557, 51700},
  {"FSALConditions", 925683, 51703},
  {"FSALButcherTable", 925796, 51706},
  {"EmbeddedButcherTable", 925916, 51709},
  {"HigherOrder", 926033, 51712},
  {"LowerOrder", 926137, 51715},
  {"IController", 926241, 51718},
  {"s:2", 926338, 51721},
  {"s:3", 926417, 51724},
  {"s:4", 926497, 51727},
  {"s:5", 926578, 51730},
  {"s:6", 926659, 51733},
  {"DahlquistEquation", 926754, 51736},
  {"StiffnessDevice", 926871, 51739},
  {"StiffButcherTable", 926989, 51742},
  {"s:7", 927097, 51745},
  {"PIController", 927188, 51748},
  {"StepSizeSafetyFactors", 927307, 51751},
  {"StepSizeRatioBounds", 927433, 51754},
  {"s:8", 927541, 51757}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 56, 1, 88, "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{
Cell[1857, 58, 50, 1, 61, "Section",
  CellTags->"s:1"],
Cell[1910, 61, 98, 3, 24, "Text"],
Cell[2011, 66, 299, 7, 58, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[2335, 77, 36, 0, 36, "Subsection"],
Cell[2374, 79, 121, 3, 24, "Text"],
Cell[2498, 84, 178, 4, 30, "NumberedEquation",
  CellTags->"EulersMethod"],
Cell[2679, 90, 52, 0, 24, "Text"],
Cell[2734, 92, 308, 7, 40, "Text"],
Cell[3045, 101, 146, 6, 24, "Text"],

Cell[CellGroupData[{
Cell[3216, 111, 243, 5, 42, "Input"],
Cell[3462, 118, 295, 8, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[3806, 132, 49, 0, 36, "Subsection"],
Cell[3858, 134, 201, 4, 40, "Text"],
Cell[4062, 140, 149, 5, 24, "Text"],
Cell[4214, 147, 473, 11, 69, "NumberedEquation",
  CellTags->"ExplicitMidpointMethod"],
Cell[4690, 160, 183, 5, 24, "Text"],
Cell[4876, 167, 154, 6, 24, "Text"],

Cell[CellGroupData[{
Cell[5055, 177, 246, 5, 42, "Input"],
Cell[5304, 184, 295, 8, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[5648, 198, 41, 0, 36, "Subsection"],
Cell[5692, 200, 216, 6, 70, "Text"],
Cell[5911, 208, 247, 7, 24, "Text"],
Cell[6161, 217, 650, 16, 67, "NumberedEquation",
  CellTags->"ERKMethod"],
Cell[6814, 235, 111, 5, 24, "Text"],
Cell[6928, 242, 136, 5, 24, "Text"],
Cell[7067, 249, 146, 3, 52, "NumberedEquation",
  CellTags->"RowSumConditions"],
Cell[7216, 254, 197, 4, 40, "Text"],
Cell[7416, 260, 403, 13, 40, "Text"],
Cell[7822, 275, 201, 5, 24, "Text"],
Cell[8026, 282, 150, 2, 31, "DisplayFormula"],
Cell[8179, 286, 801, 25, 40, "Text"],
Cell[8983, 313, 783, 16, 97, "NumberedEquation",
  CellTags->"ButcherTable"],
Cell[9769, 331, 107, 3, 24, "Text"],
Cell[9879, 336, 241, 6, 40, "Text"],

Cell[CellGroupData[{
Cell[10145, 346, 32, 0, 34, "Subsubsection"],
Cell[10180, 348, 167, 5, 24, "Text"],
Cell[10350, 355, 338, 10, 70, "NumberedEquation",
  CellTags->"ExplicitMidpointButcherTable"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[10737, 371, 34, 0, 70, "Subsection"],
Cell[10774, 373, 225, 4, 70, "Text"],
Cell[11002, 379, 174, 3, 70, "NumberedEquation",
  CellTags->"FSALConditions"],
Cell[11179, 384, 165, 7, 70, "Text"],
Cell[11347, 393, 887, 18, 70, "NumberedEquation",
  CellTags->"FSALButcherTable"],
Cell[12237, 413, 311, 9, 40, "Text"],
Cell[12551, 424, 265, 7, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[12853, 436, 63, 0, 36, "Subsection"],
Cell[12919, 438, 354, 9, 40, "Text"],
Cell[13276, 449, 1038, 20, 137, "NumberedEquation",
  CellTags->"EmbeddedButcherTable"],
Cell[14317, 471, 41, 0, 24, "Text"],
Cell[14361, 473, 173, 4, 52, "NumberedEquation",
  CellTags->"HigherOrder"],
Cell[14537, 479, 188, 5, 52, "NumberedEquation",
  CellTags->"LowerOrder"],
Cell[14728, 486, 537, 20, 24, "Text"],
Cell[15268, 508, 379, 10, 40, "Text"],
Cell[15650, 520, 1261, 35, 43, "Text"],
Cell[16914, 557, 124, 2, 52, "DisplayFormula"],
Cell[17041, 561, 248, 7, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[17326, 573, 34, 0, 70, "Subsection"],
Cell[17363, 575, 93, 3, 70, "Text"],
Cell[17459, 580, 217, 5, 70, "NumberedEquation",
  CellTags->"IController"],
Cell[17679, 587, 319, 11, 70, "Text"],
Cell[18001, 600, 129, 3, 70, "Text"],
Cell[18133, 605, 430, 14, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[18600, 624, 30, 0, 70, "Subsection"],
Cell[18633, 626, 103, 3, 70, "Text"],
Cell[18739, 631, 60, 0, 70, "Text"],
Cell[18802, 633, 52, 0, 70, "BulletedList"],
Cell[18857, 635, 127, 3, 70, "BulletedList"],
Cell[18987, 640, 486, 18, 70, "BulletedList"],
Cell[19476, 660, 220, 7, 26, "BulletedList"],
Cell[19699, 669, 331, 10, 40, "Text"],
Cell[20033, 681, 350, 10, 40, "Text"],
Cell[20386, 693, 355, 8, 40, "Text"],
Cell[20744, 703, 183, 4, 40, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[20976, 713, 45, 1, 61, "Section",
  CellTags->"s:2"],
Cell[21024, 716, 102, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[21151, 723, 109, 2, 26, "Input"],
Cell[21263, 727, 834, 19, 90, "Output"]
}, Open  ]],
Cell[22112, 749, 85, 1, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[22222, 754, 131, 3, 26, "Input"],
Cell[22356, 759, 546, 15, 41, "Output"]
}, Open  ]],
Cell[22917, 777, 77, 0, 40, "MathCaption"],
Cell[22997, 779, 129, 3, 26, "Input"],
Cell[23129, 784, 52, 0, 70, "MathCaption"],

Cell[CellGroupData[{
Cell[23206, 788, 134, 3, 70, "Input"],
Cell[23343, 793, 40720, 833, 70, 5699, 395, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[64112, 1632, 55, 1, 70, "Section",
  CellTags->"s:3"],
Cell[64170, 1635, 145, 5, 70, "Text"],
Cell[64318, 1642, 101, 3, 70, "MathCaption"],

Cell[CellGroupData[{
Cell[64444, 1649, 119, 3, 70, "Input"],
Cell[64566, 1654, 152, 3, 70, "Output"]
}, Open  ]],
Cell[64733, 1660, 149, 3, 40, "Text"],

Cell[CellGroupData[{
Cell[64907, 1667, 31, 0, 36, "Subsection"],
Cell[64941, 1669, 245, 6, 40, "Text"],
Cell[65189, 1677, 156, 4, 26, "BulletedList"],
Cell[65348, 1683, 171, 5, 26, "BulletedList"],
Cell[65522, 1690, 134, 4, 40, "MathCaption"],
Cell[65659, 1696, 554, 12, 154, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[66250, 1713, 40, 0, 36, "Subsection"],
Cell[66293, 1715, 192, 4, 40, "Text"],
Cell[66488, 1721, 244, 7, 40, "Text"],
Cell[66735, 1730, 236, 5, 56, "MathCaption"],
Cell[66974, 1737, 280, 6, 74, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[67291, 1748, 47, 0, 70, "Subsection"],
Cell[67341, 1750, 202, 5, 70, "Text"],
Cell[67546, 1757, 249, 7, 70, "Text"],

Cell[CellGroupData[{
Cell[67820, 1768, 34, 0, 70, "Subsubsection"],
Cell[67857, 1770, 151, 3, 70, "Text"],
Cell[68011, 1775, 179, 5, 70, "MathCaption"],
Cell[68193, 1782, 368, 9, 70, "Input"],
Cell[68564, 1793, 126, 3, 70, "MathCaption"],
Cell[68693, 1798, 114, 2, 70, "Input"],
Cell[68810, 1802, 54, 0, 70, "MathCaption"],

Cell[CellGroupData[{
Cell[68889, 1806, 223, 4, 70, "Input"],
Cell[69115, 1812, 928, 18, 70, "Output"]
}, Open  ]],
Cell[70058, 1833, 214, 4, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[70309, 1842, 34, 0, 70, "Subsubsection"],
Cell[70346, 1844, 210, 4, 70, "Text"],
Cell[70559, 1850, 130, 3, 70, "Text"],
Cell[70692, 1855, 176, 4, 70, "Text"],
Cell[70871, 1861, 179, 5, 70, "MathCaption"],
Cell[71053, 1868, 356, 9, 70, "Input"],
Cell[71412, 1879, 163, 4, 70, "MathCaption"],
Cell[71578, 1885, 222, 5, 70, "Input"],
Cell[71803, 1892, 70, 0, 40, "MathCaption"],
Cell[71876, 1894, 173, 5, 42, "Input"],
Cell[72052, 1901, 352, 6, 72, "MathCaption"],

Cell[CellGroupData[{
Cell[72429, 1911, 357, 6, 74, "Input"],
Cell[72789, 1919, 63789, 1912, 239, 36803, 1573, "GraphicsData", \
"PostScript", "Graphics"],
Cell[136581, 3833, 155, 4, 25, "Output"]
}, Open  ]],
Cell[136751, 3840, 217, 4, 40, "Text"],
Cell[136971, 3846, 101, 3, 24, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[137133, 3856, 57, 1, 61, "Section",
  CellTags->"s:4"],
Cell[137193, 3859, 146, 4, 70, "Text"],
Cell[137342, 3865, 102, 3, 70, "Text"],
Cell[137447, 3870, 65, 0, 70, "BulletedList"],
Cell[137515, 3872, 105, 3, 70, "BulletedList"],
Cell[137623, 3877, 55, 0, 70, "BulletedList"],

Cell[CellGroupData[{
Cell[137703, 3881, 54, 0, 70, "Subsection"],
Cell[137760, 3883, 199, 5, 70, "MathCaption"],
Cell[137962, 3890, 224, 7, 70, "Input"],
Cell[138189, 3899, 199, 4, 70, "Text"],
Cell[138391, 3905, 64, 0, 70, "MathCaption"],

Cell[CellGroupData[{
Cell[138480, 3909, 272, 6, 70, "Input"],
Cell[138755, 3917, 546, 15, 70, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[139350, 3938, 27, 0, 70, "Subsection"],
Cell[139380, 3940, 98, 3, 70, "Text"],
Cell[139481, 3945, 242, 7, 70, "Text"],
Cell[139726, 3954, 109, 3, 70, "MathCaption"],
Cell[139838, 3959, 424, 9, 70, "Input"],
Cell[140265, 3970, 223, 7, 70, "Text"],
Cell[140491, 3979, 115, 3, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[140643, 3987, 42, 0, 70, "Subsection"],
Cell[140688, 3989, 244, 7, 70, "Text"],
Cell[140935, 3998, 164, 4, 70, "Text"],
Cell[141102, 4004, 109, 3, 70, "MathCaption"],
Cell[141214, 4009, 783, 14, 70, "Input"],
Cell[142000, 4025, 170, 4, 70, "Text"],
Cell[142173, 4031, 412, 10, 56, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[142622, 4046, 45, 0, 36, "Subsection"],
Cell[142670, 4048, 273, 8, 24, "Text"],
Cell[142946, 4058, 109, 3, 40, "MathCaption"],
Cell[143058, 4063, 941, 17, 234, "Input"],
Cell[144002, 4082, 269, 7, 40, "Text"],
Cell[144274, 4091, 280, 5, 72, "MathCaption"],

Cell[CellGroupData[{
Cell[144579, 4100, 270, 6, 42, "Input"],
Cell[144852, 4108, 546, 15, 41, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[145447, 4129, 39, 0, 70, "Subsection"],
Cell[145489, 4131, 90, 3, 70, "Text"],
Cell[145582, 4136, 182, 4, 70, "MathCaption"],
Cell[145767, 4142, 288, 5, 70, "Input"],
Cell[146058, 4149, 78, 0, 70, "MathCaption"],
Cell[146139, 4151, 239, 5, 70, "Input"],
Cell[146381, 4158, 104, 3, 70, "MathCaption"],
Cell[146488, 4163, 254, 5, 58, "Input"],
Cell[146745, 4170, 95, 3, 70, "MathCaption"],
Cell[146843, 4175, 251, 4, 58, "Input"],
Cell[147097, 4181, 126, 3, 40, "MathCaption"],
Cell[147226, 4186, 112, 2, 26, "Input"],
Cell[147341, 4190, 54, 0, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[147420, 4194, 222, 4, 58, "Input"],
Cell[147645, 4200, 585, 14, 102, "Output"]
}, Open  ]],
Cell[148245, 4217, 109, 3, 24, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[148403, 4226, 52, 1, 66, "Section",
  CellTags->"s:5"],
Cell[148458, 4229, 147, 3, 40, "Text"],
Cell[148608, 4234, 305, 5, 72, "MathCaption"],
Cell[148916, 4241, 152, 5, 62, "Input"],
Cell[149071, 4248, 104, 3, 40, "MathCaption"],
Cell[149178, 4253, 330, 11, 164, "Input"],
Cell[149511, 4266, 361, 6, 56, "Text"],
Cell[149875, 4274, 176, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[150076, 4282, 147, 3, 26, "Input"],
Cell[150226, 4287, 546, 15, 41, "Output"]
}, Open  ]],
Cell[150787, 4305, 213, 5, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[151037, 4315, 47, 1, 66, "Section",
  CellTags->"s:6"],
Cell[151087, 4318, 118, 3, 24, "Text"],
Cell[151208, 4323, 149, 3, 40, "Text"],
Cell[151360, 4328, 160, 4, 40, "Text"],

Cell[CellGroupData[{
Cell[151545, 4336, 38, 0, 40, "Subsection"],
Cell[151586, 4338, 123, 3, 24, "Text"],
Cell[151712, 4343, 223, 4, 30, "NumberedEquation",
  CellTags->"DahlquistEquation"],
Cell[151938, 4349, 338, 12, 24, "Text"],
Cell[152279, 4363, 270, 6, 40, "Text"],
Cell[152552, 4371, 123, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[152700, 4378, 141, 4, 26, "Input"],
Cell[152844, 4384, 126, 3, 44, "Output"]
}, Open  ]],
Cell[152985, 4390, 262, 6, 40, "Text"],
Cell[153250, 4398, 155, 3, 56, "MathCaption"],
Cell[153408, 4403, 97, 2, 26, "Input"],
Cell[153508, 4407, 174, 5, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[153707, 4416, 86, 2, 26, "Input"],
Cell[153796, 4420, 582729, 42735, 294, 567652, 42544, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[736540, 47158, 456, 13, 40, "Text"],
Cell[736999, 47173, 245, 7, 24, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[737281, 47185, 41, 0, 36, "Subsection"],
Cell[737325, 47187, 269, 8, 24, "Text"],
Cell[737597, 47197, 133, 3, 24, "Text"],
Cell[737733, 47202, 278, 5, 48, "NumberedEquation",
  CellTags->"StiffnessDevice"],
Cell[738014, 47209, 227, 11, 24, "Text"],
Cell[738244, 47222, 320, 12, 24, "Text"],
Cell[738567, 47236, 134, 3, 24, "Text"],
Cell[738704, 47241, 240, 6, 31, "Text"],
Cell[738947, 47249, 318, 12, 24, "Text"],
Cell[739268, 47263, 897, 18, 115, "NumberedEquation",
  CellTags->"StiffButcherTable"],
Cell[740168, 47283, 211, 9, 24, "Text"],
Cell[740382, 47294, 263, 6, 40, "Text"],
Cell[740648, 47302, 253, 8, 24, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[740938, 47315, 30, 0, 40, "Subsection"],
Cell[740971, 47317, 74, 0, 40, "MathCaption"],
Cell[741048, 47319, 107, 2, 26, "Input"],
Cell[741158, 47323, 105, 3, 40, "MathCaption"],
Cell[741266, 47328, 129, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[741420, 47335, 116, 2, 26, "Input"],
Cell[741539, 47339, 761, 20, 57, "Output"]
}, Open  ]],
Cell[742315, 47362, 234, 5, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[742574, 47371, 224, 5, 42, "Input"],
Cell[742801, 47378, 243, 5, 31, "Message"],
Cell[743047, 47385, 182, 4, 18, "Message"]
}, Open  ]],
Cell[743244, 47392, 143, 3, 40, "Text"],
Cell[743390, 47397, 129, 3, 24, "Text"],
Cell[743522, 47402, 191, 4, 56, "MathCaption"],

Cell[CellGroupData[{
Cell[743738, 47410, 113, 2, 26, "Input"],
Cell[743854, 47414, 154, 4, 27, "Output"]
}, Open  ]],
Cell[744023, 47421, 146, 3, 40, "Text"],
Cell[744172, 47426, 106, 3, 40, "MathCaption"],
Cell[744281, 47431, 199, 4, 42, "Input"],
Cell[744483, 47437, 99, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[744607, 47444, 222, 5, 42, "Input"],
Cell[744832, 47451, 251, 5, 31, "Message"],
Cell[745086, 47458, 914, 26, 57, "Output"]
}, Open  ]],
Cell[746015, 47487, 296, 9, 40, "Text"],
Cell[746314, 47498, 243, 5, 40, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[746606, 47509, 60, 1, 66, "Section",
  CellTags->"s:7"],
Cell[746669, 47512, 222, 6, 40, "Text"],
Cell[746894, 47520, 62, 0, 40, "MathCaption"],
Cell[746959, 47522, 107, 2, 26, "Input"],
Cell[747069, 47526, 158, 4, 56, "MathCaption"],
Cell[747230, 47532, 233, 4, 42, "Input"],
Cell[747466, 47538, 172, 5, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[747663, 47547, 172, 4, 42, "Input"],
Cell[747838, 47553, 57687, 1102, 184, 6954, 471, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[805540, 48658, 202, 4, 40, "Text"],
Cell[805745, 48664, 75, 0, 24, "Text"],
Cell[805823, 48666, 142, 3, 24, "Text"],
Cell[805968, 48671, 139, 3, 24, "Text"],

Cell[CellGroupData[{
Cell[806132, 48678, 37, 0, 36, "Subsection"],
Cell[806172, 48680, 187, 5, 24, "Text"],
Cell[806362, 48687, 148, 3, 24, "Text"],
Cell[806513, 48692, 398, 7, 52, "NumberedEquation",
  CellTags->"PIController"],
Cell[806914, 48701, 103, 3, 24, "Text"],
Cell[807020, 48706, 242, 8, 24, "Text"],
Cell[807265, 48716, 101, 2, 30, "DisplayFormula"],
Cell[807369, 48720, 410, 15, 24, "Text"],
Cell[807782, 48737, 262, 7, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[808081, 48749, 30, 0, 36, "Subsection"],

Cell[CellGroupData[{
Cell[808136, 48753, 38, 0, 34, "Subsubsection"],
Cell[808177, 48755, 200, 6, 40, "Text"],
Cell[808380, 48763, 89, 3, 24, "Text"],
Cell[808472, 48768, 269, 7, 30, "DisplayFormula"],
Cell[808744, 48777, 66, 0, 40, "MathCaption"],
Cell[808813, 48779, 311, 6, 74, "Input"],
Cell[809127, 48787, 127, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[809279, 48794, 175, 4, 42, "Input"],
Cell[809457, 48800, 25402, 715, 184, 7040, 483, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[834908, 49521, 41, 0, 34, "Subsubsection"],
Cell[834952, 49523, 307, 9, 40, "Text"],
Cell[835262, 49534, 97, 3, 40, "MathCaption"],
Cell[835362, 49539, 112, 2, 26, "Input"],

Cell[CellGroupData[{
Cell[835499, 49545, 172, 4, 42, "Input"],
Cell[835674, 49551, 40053, 892, 184, 7152, 481, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[875742, 50446, 98, 3, 40, "MathCaption"],

Cell[CellGroupData[{
Cell[875865, 50453, 175, 4, 42, "Input"],
Cell[876043, 50459, 42156, 1008, 184, 8907, 592, "GraphicsData", \
"PostScript", "Graphics"]
}, Open  ]],
Cell[918214, 51470, 194, 6, 24, "Text"],
Cell[918411, 51478, 186, 7, 26, "BulletedList"],
Cell[918600, 51487, 187, 7, 26, "BulletedList"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[918836, 51500, 33, 0, 36, "Subsection"],
Cell[918872, 51502, 292, 7, 40, "Text"],
Cell[919167, 51511, 445, 15, 40, "Text"],
Cell[919615, 51528, 443, 7, 52, "NumberedEquation",
  CellTags->"StepSizeSafetyFactors"],
Cell[920061, 51537, 225, 8, 24, "Text"],
Cell[920289, 51547, 354, 10, 24, "Text"],
Cell[920646, 51559, 199, 3, 47, "NumberedEquation",
  CellTags->"StepSizeRatioBounds"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[920894, 51568, 52, 1, 61, "Section",
  CellTags->"s:8"],
Cell[920949, 51571, 2934, 75, 328, "DefinitionBox3Col"],
Cell[923886, 51648, 105, 4, 15, "Caption"]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

