(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.0'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     76673,       2443]*)
(*NotebookOutlinePosition[     77936,       2491]*)
(*  CellTagsIndexPosition[     77794,       2482]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["FixedStep", "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"s:1"],

Cell["\<\
It is often useful to carry out a numerical integration using fixed \
step sizes.\
\>", "Text"],

Cell[TextData[{
  "For example, certain methods such as ",
  StyleBox["DoubleStep", "MR"],
  " and ",
  StyleBox["Extrapolation", "MR"],
  " carry out a sequence of fixed-step integrations before combining the \
solutions to obtain a more accurate method with an error estimate that allows \
adaptive step sizes to be taken."
}], "Text"],

Cell[TextData[{
  "The method ",
  StyleBox["FixedStep", "MR"],
  " allows any one-step integration method to be invoked using fixed step \
sizes."
}], "Text"],

Cell["\<\
This loads a package with some example problems and a package with \
some utility functions.\
\>", "Text"],

Cell[BoxData[{
    \(\(Needs["\<DifferentialEquations`NDSolveProblems`\>"];\)\), "\
\[IndentingNewLine]", 
    \(\(Needs["\<DifferentialEquations`NDSolveUtilities`\>"];\)\)}], "Input",
  CellLabel->"In[4]:=",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
  CellTags->"s:2"],

Cell["Define an example problem.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(system\  = \ GetNDSolveProblem["\<BrusselatorODE\>"]\)], "Input",
  CellLabel->"In[2]:="],

Cell[BoxData[
    RowBox[{"NDSolveProblem", "[", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{"{", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_1\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(1 - 4\ Y\_1[T] + Y\_1[T]\^2\ Y\_2[T]\)}], ",", 
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_2\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(3\ Y\_1[T] - Y\_1[T]\^2\ Y\_2[T]\)}]}], "}"}], 
          ",", \({Y\_1[0] \[Equal] 3\/2, Y\_2[0] \[Equal] 3}\), 
          ",", \({Y\_1[T], Y\_2[T]}\), ",", \({T, 0, 20}\), ",", \({}\), 
          ",", \({}\)}], "}"}], "]"}]], "Output",
  CellLabel->"Out[2]="]
}, Open  ]],

Cell[TextData[{
  "This integrates a differential system using the method ",
  StyleBox["ExplicitEuler", "MR"],
  " with a fixed step size of",
  " ",
  Cell[BoxData[
      \(TraditionalForm\`1/10\)]],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[{\(y''\)[t] \[Equal] \(-y[t]\), 
        y[0] \[Equal] 1, \(y'\)[0] \[Equal] 0}, y, {t, 0, 1}, 
      StartingStepSize \[Rule] 1/10, \ 
      Method \[Rule] {"\<FixedStep\>", \ 
          Method \[Rule] "\<ExplicitEuler\>"}]\)], "Input",
  CellLabel->"In[1]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{"y", "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
            False,
            Editable->False]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[1]="]
}, Open  ]],

Cell[TextData[{
  "Actually the ",
  StyleBox["ExplicitEuler", "MR"],
  " method has no adaptive step size control. Therefore, the integration is \
already carried out using fixed step sizes so the specification of ",
  StyleBox["FixedStep", "MR"],
  " is unnecessary."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(sol\  = \ 
        NDSolve[system, 
          StartingStepSize \[Rule] 1/10, \ \ Method \[Rule] 
            ExplicitEuler];\)\), "\[IndentingNewLine]", 
    \(\(StepDataPlot[sol, PlotRange \[Rule] {0, \ 0.2}];\)\)}], "Input",
  CellLabel->"In[55]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0478583 1.61546e-17 3.09017 [
[.2631 -0.0125 -3 -9 ]
[.2631 -0.0125 3 0 ]
[.50239 -0.0125 -6 -9 ]
[.50239 -0.0125 6 0 ]
[.74168 -0.0125 -6 -9 ]
[.74168 -0.0125 6 0 ]
[.98098 -0.0125 -6 -9 ]
[.98098 -0.0125 6 0 ]
[.01131 .15451 -24 -4.5 ]
[.01131 .15451 0 4.5 ]
[.01131 .30902 -18 -4.5 ]
[.01131 .30902 0 4.5 ]
[.01131 .46353 -24 -4.5 ]
[.01131 .46353 0 4.5 ]
[.01131 .61803 -18 -4.5 ]
[.01131 .61803 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.2631 0 m
.2631 .00625 L
s
[(5)] .2631 -0.0125 0 1 Mshowa
.50239 0 m
.50239 .00625 L
s
[(10)] .50239 -0.0125 0 1 Mshowa
.74168 0 m
.74168 .00625 L
s
[(15)] .74168 -0.0125 0 1 Mshowa
.98098 0 m
.98098 .00625 L
s
[(20)] .98098 -0.0125 0 1 Mshowa
.125 Mabswid
.07167 0 m
.07167 .00375 L
s
.11953 0 m
.11953 .00375 L
s
.16738 0 m
.16738 .00375 L
s
.21524 0 m
.21524 .00375 L
s
.31096 0 m
.31096 .00375 L
s
.35882 0 m
.35882 .00375 L
s
.40668 0 m
.40668 .00375 L
s
.45453 0 m
.45453 .00375 L
s
.55025 0 m
.55025 .00375 L
s
.59811 0 m
.59811 .00375 L
s
.64597 0 m
.64597 .00375 L
s
.69383 0 m
.69383 .00375 L
s
.78954 0 m
.78954 .00375 L
s
.8374 0 m
.8374 .00375 L
s
.88526 0 m
.88526 .00375 L
s
.93312 0 m
.93312 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
.02381 .15451 m
.03006 .15451 L
s
[(0.05)] .01131 .15451 1 0 Mshowa
.02381 .30902 m
.03006 .30902 L
s
[(0.1)] .01131 .30902 1 0 Mshowa
.02381 .46353 m
.03006 .46353 L
s
[(0.15)] .01131 .46353 1 0 Mshowa
.02381 .61803 m
.03006 .61803 L
s
[(0.2)] .01131 .61803 1 0 Mshowa
.125 Mabswid
.02381 .0309 m
.02756 .0309 L
s
.02381 .0618 m
.02756 .0618 L
s
.02381 .09271 m
.02756 .09271 L
s
.02381 .12361 m
.02756 .12361 L
s
.02381 .18541 m
.02756 .18541 L
s
.02381 .21631 m
.02756 .21631 L
s
.02381 .24721 m
.02756 .24721 L
s
.02381 .27812 m
.02756 .27812 L
s
.02381 .33992 m
.02756 .33992 L
s
.02381 .37082 m
.02756 .37082 L
s
.02381 .40172 m
.02756 .40172 L
s
.02381 .43262 m
.02756 .43262 L
s
.02381 .49443 m
.02756 .49443 L
s
.02381 .52533 m
.02756 .52533 L
s
.02381 .55623 m
.02756 .55623 L
s
.02381 .58713 m
.02756 .58713 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .30902 m
.0286 .30902 L
.03338 .30902 L
.03817 .30902 L
.04295 .30902 L
.04774 .30902 L
.05252 .30902 L
.05731 .30902 L
.0621 .30902 L
.06688 .30902 L
.07167 .30902 L
.07645 .30902 L
.08124 .30902 L
.08603 .30902 L
.09081 .30902 L
.0956 .30902 L
.10038 .30902 L
.10517 .30902 L
.10995 .30902 L
.11474 .30902 L
.11953 .30902 L
.12431 .30902 L
.1291 .30902 L
.13388 .30902 L
.13867 .30902 L
.14346 .30902 L
.14824 .30902 L
.15303 .30902 L
.15781 .30902 L
.1626 .30902 L
.16738 .30902 L
.17217 .30902 L
.17696 .30902 L
.18174 .30902 L
.18653 .30902 L
.19131 .30902 L
.1961 .30902 L
.20089 .30902 L
.20567 .30902 L
.21046 .30902 L
.21524 .30902 L
.22003 .30902 L
.22481 .30902 L
.2296 .30902 L
.23439 .30902 L
.23917 .30902 L
.24396 .30902 L
.24874 .30902 L
.25353 .30902 L
.25832 .30902 L
Mistroke
.2631 .30902 L
.26789 .30902 L
.27267 .30902 L
.27746 .30902 L
.28224 .30902 L
.28703 .30902 L
.29182 .30902 L
.2966 .30902 L
.30139 .30902 L
.30617 .30902 L
.31096 .30902 L
.31575 .30902 L
.32053 .30902 L
.32532 .30902 L
.3301 .30902 L
.33489 .30902 L
.33967 .30902 L
.34446 .30902 L
.34925 .30902 L
.35403 .30902 L
.35882 .30902 L
.3636 .30902 L
.36839 .30902 L
.37318 .30902 L
.37796 .30902 L
.38275 .30902 L
.38753 .30902 L
.39232 .30902 L
.3971 .30902 L
.40189 .30902 L
.40668 .30902 L
.41146 .30902 L
.41625 .30902 L
.42103 .30902 L
.42582 .30902 L
.43061 .30902 L
.43539 .30902 L
.44018 .30902 L
.44496 .30902 L
.44975 .30902 L
.45453 .30902 L
.45932 .30902 L
.46411 .30902 L
.46889 .30902 L
.47368 .30902 L
.47846 .30902 L
.48325 .30902 L
.48804 .30902 L
.49282 .30902 L
.49761 .30902 L
Mistroke
.50239 .30902 L
.50718 .30902 L
.51196 .30902 L
.51675 .30902 L
.52154 .30902 L
.52632 .30902 L
.53111 .30902 L
.53589 .30902 L
.54068 .30902 L
.54547 .30902 L
.55025 .30902 L
.55504 .30902 L
.55982 .30902 L
.56461 .30902 L
.56939 .30902 L
.57418 .30902 L
.57897 .30902 L
.58375 .30902 L
.58854 .30902 L
.59332 .30902 L
.59811 .30902 L
.6029 .30902 L
.60768 .30902 L
.61247 .30902 L
.61725 .30902 L
.62204 .30902 L
.62682 .30902 L
.63161 .30902 L
.6364 .30902 L
.64118 .30902 L
.64597 .30902 L
.65075 .30902 L
.65554 .30902 L
.66033 .30902 L
.66511 .30902 L
.6699 .30902 L
.67468 .30902 L
.67947 .30902 L
.68425 .30902 L
.68904 .30902 L
.69383 .30902 L
.69861 .30902 L
.7034 .30902 L
.70818 .30902 L
.71297 .30902 L
.71776 .30902 L
.72254 .30902 L
.72733 .30902 L
.73211 .30902 L
.7369 .30902 L
Mistroke
.74168 .30902 L
.74647 .30902 L
.75126 .30902 L
.75604 .30902 L
.76083 .30902 L
.76561 .30902 L
.7704 .30902 L
.77519 .30902 L
.77997 .30902 L
.78476 .30902 L
.78954 .30902 L
.79433 .30902 L
.79911 .30902 L
.8039 .30902 L
.80869 .30902 L
.81347 .30902 L
.81826 .30902 L
.82304 .30902 L
.82783 .30902 L
.83262 .30902 L
.8374 .30902 L
.84219 .30902 L
.84697 .30902 L
.85176 .30902 L
.85654 .30902 L
.86133 .30902 L
.86612 .30902 L
.8709 .30902 L
.87569 .30902 L
.88047 .30902 L
.88526 .30902 L
.89005 .30902 L
.89483 .30902 L
.89962 .30902 L
.9044 .30902 L
.90919 .30902 L
.91397 .30902 L
.91876 .30902 L
.92355 .30902 L
.92833 .30902 L
.93312 .30902 L
.9379 .30902 L
.94269 .30902 L
.94748 .30902 L
.95226 .30902 L
.95705 .30902 L
.96183 .30902 L
.96662 .30902 L
.9714 .30902 L
.97619 .30902 L
Mistroke
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[55]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool005Yoo`<003Qoo`@000=oo`8003Eoo`@0009oo`<003Eoo`@000=oo`8000Ioo`00GGoo00<007oo
Ool0=goo00D007ooOomoo`00009oo`03001oogoo03Aoo`03001oogoo00Aoo`03001oogoo039oo`03
001oogoo00=oo`04001oogoo0005Ool005eoo`03001oogoo03Moo`05001oogooOol00002Ool00`00
Oomoo`0dOol00`00Oomoo`04Ool00`00Oomoo`0cOol00`00Oomoo`02Ool01000Oomoo`001Goo001J
Ool3000jOol01@00Oomoogoo00000Woo00<007ooOol0=7oo00@007ooOomoo`<003Moo`05001oogoo
Ool00002Ool00`00Oomoo`03Ool005Yoo`03001oogoo03Yoo`05001oogooOol00002Ool00`00Oomo
o`0dOol01@00Oomoogoo0000>Woo00@007ooOol0009oo`03001oogoo00=oo`00FWoo00<007ooOol0
>7oo0`000goo00@007ooOol003Aoo`<000=oo`03001oogoo03Eoo`04001oogoo0002Ool01000Oomo
o`001Goo001JOol4000iOol00`00Oomoo`02Ool2000gOol01000Oomoogoo1000=Goo0P0017oo0P00
1Woo003oOolQOol00?moob5oo`00ogoo8Goo000GOooo00060004Ool001ioo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Moo`00
7Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007oo
Ool0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol0
0`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3o
Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`80
0?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`00
7Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007oo
Ool0ogoo000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol0
0`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3o
Ool001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001o
ogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`00
17oo0P0017oo00<007ooOol00goo0P000goo0`001Woo00<007ooOol0ogoo0003Ool01000Oomoo`00
27oo00@007ooOol000Eoo`03001oogoo00=oo`03001oogoo0?moo`000goo00@007ooOol000Qoo`04
001oogoo0005Ool00`00Oomoo`03Ool2003oOol1Ool000=oo`04001oogoo0008Ool01000Oomoo`00
0Woo0`001Woo00<007ooOol0ogoo0003Ool01000Oomoo`0027oo00@007ooOol0009oo`03001oogoo
00Ioo`03001oogoo0?moo`000goo00@007ooOol000Qoo`04001oogoo0002Ool00`00Oomoo`06Ool0
0`00Oomoo`3oOol000Aoo`8000Yoo`8000=oo`@000Eoo`03001oogoo0?moo`007Woo00<007ooOol0
ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`800?mo
o`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo
00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0
ogoo000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001oogoo
0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`002Woo
0P0017oo00<007ooOol00Woo10001Goo00<007ooOol0ogoo0009Ool01000Oomoo`002Woo00<007oo
Ool017oo00<007ooOol0ogoo0009Ool01000Oomoo`002Woo00<007ooOol017oom`002goo0009Ool0
1000Oomoo`002Woo00<007ooOol017oo00<007ooOol0ogoo0009Ool01000Oomoo`002Woo00<007oo
Ool017oo00<007ooOol0ogoo0009Ool01000Oomoo`0027oo0`001Woo00<007ooOol0ogoo000:Ool2
000;Ool00`00Oomoo`04Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0
ogoo000NOol00`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol2003oOol1Ool001ioo`03001oogoo
0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo
00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo
000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo
0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol000Aoo`8000Aoo`03001oogoo009o
o`@0009oo`<000Ioo`03001oogoo0?moo`000goo00@007ooOol000Yoo`03001oogoo00Aoo`03001o
ogoo00=oo`03001oogoo0?moo`000goo00@007ooOol000Yoo`03001oogoo00Aoo`03001oogoo00=o
o`800?moo`5oo`000goo00@007ooOol000Yoo`04001oogooOol30006Ool00`00Oomoo`3oOol000=o
o`04001oogoo000:Ool01@00Oomoogoo000027oo00<007ooOol0ogoo0003Ool01000Oomoo`0027oo
0`000goo00<007ooOol01Woo00<007ooOol0ogoo0004Ool2000;Ool01000Oomoogoo10001Goo00<0
07ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo
000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomo
o`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001io
o`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?mo
o`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<0
07ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000N
Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomo
o`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol2003oOol1Ool001ioo`03
001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?mo
o`007Woo00<007ooOol0ogoo000:Ool20004Ool00`00Oomoo`02Ool40005Ool00`00Oomoo`3oOol0
00Uoo`04001oogoo0008Ool00`00Oomoo`06Ool00`00Oomoo`3oOol000Uoo`04001oogoo0009Ool0
0`00Oomoo`05Ool2003oOol1Ool000Uoo`04001oogoo000:Ool00`00Oomoo`3oOol7Ool000Uoo`04
001oogoo000;Ool00`00Oomoo`3oOol6Ool000Uoo`04001oogoo0008Ool01000Oomoo`00ogoo27oo
000:Ool2000:Ool2003oOol9Ool00001\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {265.062, 88.125}} -> {-8.45975, \
0.0947386, 0.0806123, 0.00124846}}]
}, Open  ]],

Cell[TextData[{
  "Here are the step sizes taken by the method ",
  StyleBox["ExplicitRungeKutta", "MR"],
  " for this problem."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(sol\  = \ 
        NDSolve[system, StartingStepSize \[Rule] 1/10, \ 
          Method \[Rule] "\<ExplicitRungeKutta\>"];\)\), \
"\[IndentingNewLine]", 
    \(\(StepDataPlot[sol];\)\)}], "Input",
  CellLabel->"In[42]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0483251 -0.0285591 1.07619 [
[.26543 .06656 -3 -9 ]
[.26543 .06656 3 0 ]
[.50706 .06656 -6 -9 ]
[.50706 .06656 6 0 ]
[.74869 .06656 -6 -9 ]
[.74869 .06656 6 0 ]
[.99031 .06656 -6 -9 ]
[.99031 .06656 6 0 ]
[.01131 .18668 -18 -4.5 ]
[.01131 .18668 0 4.5 ]
[.01131 .2943 -18 -4.5 ]
[.01131 .2943 0 4.5 ]
[.01131 .40192 -18 -4.5 ]
[.01131 .40192 0 4.5 ]
[.01131 .50954 -18 -4.5 ]
[.01131 .50954 0 4.5 ]
[.01131 .61715 -18 -4.5 ]
[.01131 .61715 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.26543 .07906 m
.26543 .08531 L
s
[(5)] .26543 .06656 0 1 Mshowa
.50706 .07906 m
.50706 .08531 L
s
[(10)] .50706 .06656 0 1 Mshowa
.74869 .07906 m
.74869 .08531 L
s
[(15)] .74869 .06656 0 1 Mshowa
.99031 .07906 m
.99031 .08531 L
s
[(20)] .99031 .06656 0 1 Mshowa
.125 Mabswid
.07213 .07906 m
.07213 .08281 L
s
.12046 .07906 m
.12046 .08281 L
s
.16878 .07906 m
.16878 .08281 L
s
.21711 .07906 m
.21711 .08281 L
s
.31376 .07906 m
.31376 .08281 L
s
.36209 .07906 m
.36209 .08281 L
s
.41041 .07906 m
.41041 .08281 L
s
.45874 .07906 m
.45874 .08281 L
s
.55539 .07906 m
.55539 .08281 L
s
.60371 .07906 m
.60371 .08281 L
s
.65204 .07906 m
.65204 .08281 L
s
.70036 .07906 m
.70036 .08281 L
s
.79701 .07906 m
.79701 .08281 L
s
.84534 .07906 m
.84534 .08281 L
s
.89366 .07906 m
.89366 .08281 L
s
.94199 .07906 m
.94199 .08281 L
s
.25 Mabswid
0 .07906 m
1 .07906 L
s
.02381 .18668 m
.03006 .18668 L
s
[(0.2)] .01131 .18668 1 0 Mshowa
.02381 .2943 m
.03006 .2943 L
s
[(0.3)] .01131 .2943 1 0 Mshowa
.02381 .40192 m
.03006 .40192 L
s
[(0.4)] .01131 .40192 1 0 Mshowa
.02381 .50954 m
.03006 .50954 L
s
[(0.5)] .01131 .50954 1 0 Mshowa
.02381 .61715 m
.03006 .61715 L
s
[(0.6)] .01131 .61715 1 0 Mshowa
.125 Mabswid
.02381 .10058 m
.02756 .10058 L
s
.02381 .12211 m
.02756 .12211 L
s
.02381 .14363 m
.02756 .14363 L
s
.02381 .16515 m
.02756 .16515 L
s
.02381 .2082 m
.02756 .2082 L
s
.02381 .22973 m
.02756 .22973 L
s
.02381 .25125 m
.02756 .25125 L
s
.02381 .27277 m
.02756 .27277 L
s
.02381 .31582 m
.02756 .31582 L
s
.02381 .33734 m
.02756 .33734 L
s
.02381 .35887 m
.02756 .35887 L
s
.02381 .38039 m
.02756 .38039 L
s
.02381 .42344 m
.02756 .42344 L
s
.02381 .44496 m
.02756 .44496 L
s
.02381 .46649 m
.02756 .46649 L
s
.02381 .48801 m
.02756 .48801 L
s
.02381 .53106 m
.02756 .53106 L
s
.02381 .55258 m
.02756 .55258 L
s
.02381 .57411 m
.02756 .57411 L
s
.02381 .59563 m
.02756 .59563 L
s
.02381 .05754 m
.02756 .05754 L
s
.02381 .03601 m
.02756 .03601 L
s
.02381 .01449 m
.02756 .01449 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
.5 Mabswid
.02381 .07906 m
.02864 .0582 L
.03254 .07025 L
.03697 .06632 L
.04124 .07897 L
.04606 .07007 L
.05049 .07747 L
.05525 .09979 L
.06102 .09533 L
.06658 .11697 L
.07312 .11926 L
.07975 .151 L
.08782 .14825 L
.09576 .17714 L
.10499 .20195 L
.11534 .24009 L
.12741 .27265 L
.14093 .23656 L
.15284 .26186 L
.16588 .30499 L
.18085 .28409 L
.19489 .30745 L
.20998 .32735 L
.22596 .37822 L
.24423 .60332 L
.2726 .36053 L
.29008 .47468 L
.31267 .3322 L
.32887 .22907 L
.34044 .14685 L
.34832 .11184 L
.35462 .09945 L
.36037 .08127 L
.3653 .04172 L
.36846 .03663 L
.37138 .02821 L
.37393 .02446 L
.37631 .02786 L
.37885 .01472 L
.38079 .02136 L
.38303 .02798 L
.38557 .02085 L
.38779 .02688 L
.39028 .02858 L
.39285 .0349 L
.39569 .04147 L
.39884 .05231 L
.40247 .07185 L
.40698 .07726 L
.41173 .09638 L
Mistroke
.41734 .1159 L
.42383 .13355 L
.43111 .18552 L
.44072 .1739 L
.44981 .23676 L
.46173 .2215 L
.47295 .26827 L
.48628 .23225 L
.49799 .26822 L
.51132 .30139 L
.52614 .27381 L
.53971 .30528 L
.5547 .32225 L
.57046 .3744 L
.58855 .59189 L
.61641 .3595 L
.63384 .46364 L
.65594 .34193 L
.67258 .25787 L
.68544 .14916 L
.69342 .11468 L
.69985 .10156 L
.70569 .08269 L
.71069 .04325 L
.71391 .03237 L
.71665 .02792 L
.71918 .02264 L
.72148 .02668 L
.72396 .01842 L
.72607 .02066 L
.72828 .03045 L
.73093 .02113 L
.73316 .02727 L
.73567 .02835 L
.73823 .03448 L
.74106 .04056 L
.74416 .05075 L
.74772 .06854 L
.75208 .07734 L
.75684 .09305 L
.7623 .11247 L
.76863 .13048 L
.77577 .17785 L
.78504 .1719 L
.79404 .2312 L
.80571 .22059 L
.8169 .30798 L
.83201 .22721 L
.84349 .27672 L
.8572 .30316 L
Mistroke
.8721 .27729 L
.88583 .30738 L
.90091 .32505 L
.91679 .37781 L
.93504 .57337 L
.96207 .28591 L
.97619 .28591 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[42]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`005Goo00<007ooOol0ogoo2Goo000EOol00`00Oomoo`3oOol9Ool001Eoo`03001o
ogoo0?moo`Uoo`005Goo00<007ooOol0ogoo2Goo000EOol2001MOol00`00Oomoo`2YOol001Eoo`03
001oogoo05aoo`03001oo`0005Yoo`8004eoo`005Goo00<007ooOol0Fgoo1000FGoo1000C7oo000E
Ool00`00Oomoo`1JOol01P00Ool00000Ool005Moo`800004Ool00000001;Ool001Eoo`03001oogoo
05Yoo`03001oogoo009oo`03001oogoo05Eoo`06001oogoo001oo`00Bgoo000EOol00`00Oomoo`1I
Ool00`00Oomoo`04Ool00`00Oomoo`1COol00`00Oomoo`04Ool00`00Oomoo`18Ool001Eoo`8005Yo
o`03001oogoo00Aoo`03001oogoo05=oo`03001oogoo00Aoo`03001oogoo04Qoo`005Goo00<007oo
Ool0?7oo0`006Woo00<007ooOol01Goo00<007ooOol05Woo10000goo0P00<Woo00<007ooOol00goo
10000Woo0`00>7oo10000goo0P000goo000EOol00`00Oomoo`0oOol00`00Oomoo`0GOol00`00Oomo
o`05Ool00`00Oomoo`0HOol01@00Oomoogoo00000Woo00<007ooOol0;goo00<007ooOol01Goo0P00
1Goo00<007ooOol0=Goo00<007ooOol00goo00@007ooOol0009oo`005Goo00<007ooOol0?goo00<0
07ooOol05goo00<007ooOol01Woo00<007ooOol05goo00D007ooOomoo`00009oo`03001oogoo02mo
o`03001oogoo00Eoo`8000Eoo`03001oogoo03Ioo`03001oogoo009oo`04001oogoo0002Ool001Eo
o`03001oogoo03aoo`<001Uoo`03001oogoo00Moo`03001oogoo01Moo`05001oogooOol00002Ool0
0`00Oomoo`0_Ool00`00Oomoo`05Ool20002Ool3000jOol01@00Oomoogoo00000Woo0@000Goo0Goo
000EOol2000mOol00`00Oomoo`0IOol00`00Oomoo`07Ool00`00Oomoo`0GOol01@00Oomoogoo0000
0Woo00<007ooOol0;goo00<007ooOol01Goo0P000Woo00<007ooOol0>goo00@007ooOol0009oo`40
005oo`5oo`005Goo0P00?Goo00<007ooOol06Goo00<007ooOol01goo00<007ooOol05Goo0`000goo
00@007ooOol0035oo`03001oogoo00=oo`@0009oo`03001oogoo03Qoo`04001oogoo0002Ool01000
Oomoo`000Woo000EOol2000mOol4000HOol00`00Oomoo`08Ool00`00Oomoo`0FOol00`00Oomoo`02
Ool2000bOol00`00Oomoo`05Ool01000Ool007oo1000>7oo0P0017oo0P000goo000EOol4001GOol0
0`00Oomoo`08Ool00`00Oomoo`1>Ool00`00Oomoo`08Ool00`00Oomoo`16Ool001Eoo`07001oo`00
Ool007oo0000E7oo00<007ooOol027oo00<007ooOol0CWoo00<007ooOol02Goo00<007ooOol0AGoo
000EOol01`00Oomoogoo001oo`0005=oo`03001oogoo00Uoo`03001oogoo04ioo`03001oogoo00Uo
o`03001oogoo04Eoo`003Wooo`003`0017oo000EOol00`00Oomoo`04Ool00`00Oomoo`03Ool00`00
Oomoo`0:Ool00`00Oomoo`0:Ool00`00Oomoo`0:Ool00`00Oomoo`0:Ool00`00Oomoo`0:Ool00`00
Oomoo`09Ool2000;Ool2000<Ool00`00Oomoo`0:Ool00`00Oomoo`0:Ool00`00Oomoo`0:Ool00`00
Oomoo`0:Ool00`00Oomoo`0:Ool2000;Ool2000;Ool00`00Oomoo`0:Ool00`00Oomoo`0:Ool00`00
Oomoo`0:Ool00`00Oomoo`0:Ool00`00Oomoo`04Ool001Eoo`03001oogoo00Aoo`03001oogoo03Mo
o`03001oogoo01Ioo`03001oogoo00Yoo`03001oogoo01Qoo`03001oogoo039oo`03001oogoo00Uo
o`8003moo`03001oogoo00Aoo`005Goo00<007ooOol017oo00<007ooOol0D7oo00<007ooOol02goo
00<007ooOol0Bgoo00<007ooOol037oo00<007ooOol0@goo000EOol00`00Oomoo`05Ool00`00Ool0
001>Ool00`00Oomoo`0<Ool00`00Oomoo`1;Ool00`00Oomoo`0<Ool00`00Oomoo`13Ool001Eoo`03
001oogoo00Eoo`<004ioo`03001oogoo00aoo`03001oogoo04Yoo`03001oogoo00eoo`03001oogoo
04=oo`005Goo0P001Woo00<007oo0000CWoo00<007ooOol037oo00<007ooOol0BWoo00<007ooOol0
3Goo00<007ooOol0@goo000EOol00`00Oomoo`07Ool00`00Oomoo`1;Ool00`00Oomoo`0>Ool00`00
Oomoo`19Ool00`00Oomoo`0>Ool00`00Oomoo`12Ool001Eoo`03001oogoo00Qoo`03001oogoo04Yo
o`03001oogoo00ioo`03001oogoo04Uoo`03001oogoo00ioo`03001oogoo049oo`005Goo00<007oo
Ool027oo00<007ooOol0BGoo00<007ooOol047oo00<007ooOol0Agoo00<007ooOol03goo00<007oo
Ool0@Woo000EOol00`00Oomoo`08Ool00`00Oomoo`19Ool00`00Oomoo`0@Ool00`00Oomoo`17Ool0
0`00Oomoo`0?Ool00`00Oomoo`12Ool001Eoo`03001oogoo00Uoo`8004Uoo`03001oogoo011oo`03
001oogoo04Moo`03001oogoo011oo`03001oogoo045oo`005Goo0P002goo00<007ooOol0AWoo00<0
07ooOol04Goo00<007ooOol0Agoo00<007ooOol047oo00<007ooOol0@Goo000EOol00`00Oomoo`0:
Ool00`00Oomoo`16Ool00`00Oomoo`0BOol00`00Oomoo`15Ool00`00Oomoo`0BOol00`00Oomoo`10
Ool001Eoo`03001oogoo00]oo`03001oogoo04Eoo`03001oogoo019oo`03001oogoo04Eoo`03001o
ogoo019oo`03001oogoo041oo`005Goo00<007ooOol02goo00<007ooOol0AGoo00<007ooOol04Woo
00<007ooOol0AGoo00<007ooOol04Woo00<007ooOol0@7oo000EOol00`00Oomoo`0;Ool00`00Oomo
o`14Ool00`00Oomoo`0COol00`00Oomoo`15Ool00`00Oomoo`0BOol00`00Oomoo`10Ool001Eoo`80
00aoo`03001oogoo04Aoo`03001oogoo01=oo`03001oogoo04Aoo`03001oogoo01=oo`03001oogoo
041oo`005Goo00<007ooOol037oo00<007ooOol0@goo00<007ooOol04goo00<007ooOol0A7oo00<0
07ooOol057oo00<007ooOol0?goo000EOol00`00Oomoo`0<Ool30013Ool00`00Oomoo`0DOol00`00
Oomoo`13Ool00`00Oomoo`0DOol00`00Oomoo`0oOol001Eoo`03001oogoo00ioo`03001oogoo045o
o`03001oogoo01Aoo`03001oogoo04=oo`03001oogoo01Aoo`03001oogoo03moo`005Goo00<007oo
Ool03goo00<007ooOol0@7oo00<007ooOol057oo00<007ooOol0@goo00<007ooOol057oo00<007oo
Ool0?goo000EOol00`00Oomoo`0?Ool00`00Oomoo`0oOol00`00Oomoo`0EOol00`00Oomoo`13Ool0
0`00Oomoo`0DOol00`00Oomoo`0oOol001Eoo`80011oo`03001oogoo03moo`03001oogoo01Eoo`03
001oogoo049oo`03001oogoo01Eoo`03001oogoo03moo`005Goo00<007ooOol03goo00<007ooOol0
?goo00<007ooOol05Goo00<007ooOol0@Woo00<007ooOol05Woo00<007ooOol0?Woo000EOol00`00
Oomoo`0@Ool00`00Oomoo`0nOol00`00Oomoo`0EOol00`00Oomoo`12Ool00`00Oomoo`0FOol00`00
Ool0000nOol001Eoo`03001oogoo011oo`03001oogoo03ioo`03001oogoo01Ioo`04001oogoo0010
Ool00`00Oomoo`0FOol3000nOol0009oo`8000Aoo`03001oogoo009oo`@000Aoo`03001oogoo011o
o`03001oogoo03ioo`03001oogoo01Ioo`04001oo`000010Ool00`00Oomoo`0FOol00`00Ool0000n
Ool00005Ool007ooOol00008Ool00`00Oomoo`05Ool00`00Oomoo`0AOol00`00Oomoo`0mOol00`00
Oomoo`0FOol200000goo001oo`0oOol00`00Oomoo`0IOol00`00Oomoo`0kOol00005Ool007ooOol0
0009Ool00`00Oomoo`04Ool3000AOol00`00Oomoo`0mOol00`00Oomoo`0FOol01000Oomoo`00@7oo
00<007ooOol06Goo00<007ooOol0>goo00001Goo001oogoo00002Woo00<007ooOol00goo00<007oo
Ool04Woo00<007ooOol0>goo00<007ooOol06Woo00<007ooOol0?Goo00<007ooOol06Woo00<007oo
Ool0>goo00001Goo001oogoo00002goo00<007ooOol00Woo00<007ooOol04Woo00<007ooOol0>goo
00<007ooOol06goo00<007ooOol0?7oo00<007ooOol06Woo00<007ooOol0>goo00001Goo001oogoo
000027oo00@007ooOol000Aoo`03001oogoo01=oo`03001oogoo03Yoo`03001oogoo01]oo`03001o
ogoo03aoo`03001oogoo01Yoo`03001oogoo03]oo`000Woo0P002Woo0P001Goo00<007ooOol04goo
00<007ooOol0>Woo00<007ooOol06goo00<007ooOol0?7oo00<007ooOol06Woo00<007ooOol0>goo
000EOol00`00Oomoo`0COol00`00Oomoo`0jOol00`00Oomoo`0KOol00`00Oomoo`0lOol00`00Oomo
o`0KOol00`00Oomoo`0jOol001Eoo`8001Aoo`03001oogoo03Yoo`03001oogoo01]oo`03001oogoo
03aoo`03001oogoo01]oo`03001oogoo03Yoo`005Goo00<007ooOol057oo00<007ooOol0>Goo00<0
07ooOol06goo00<007ooOol0?7oo00<007ooOol06goo00<007ooOol0>Woo000EOol00`00Oomoo`0D
Ool00`00Oomoo`0hOol00`00Oomoo`0LOol00`00Oomoo`0lOol00`00Oomoo`0KOol00`00Oomoo`0j
Ool001Eoo`03001oogoo01Aoo`03001oogoo03Qoo`03001oogoo01aoo`03001oogoo009oo`03001o
ogoo03Ioo`03001oogoo01aoo`05001oogooOol0000hOol001Eoo`03001oogoo01Aoo`03001oogoo
03Qoo`03001oogoo01eoo`03001oogoo008003Qoo`03001oogoo01eoo`04001oo`00000hOol001Eo
o`03001oogoo01Aoo`03001oogoo03Qoo`03001oogoo01eoo`05001oo`00Ool0000hOol00`00Oomo
o`0MOol200000goo001oo`05Ool00`00Oomoo`0_Ool001Eoo`8001Ioo`03001oogoo03Moo`03001o
ogoo01eoo`8000=oo`03001oogoo009oo`03001oogoo031oo`03001oogoo01eoo`04001oogoo0006
Ool00`00Oomoo`0_Ool001Eoo`03001oogoo01Eoo`03001oogoo00Aoo`03001oogoo031oo`03001o
ogoo01eoo`03001oogoo009oo`03001oogoo009oo`03001oogoo031oo`03001oogoo021oo`03001o
ogoo00Aoo`03001oogoo02moo`005Goo00<007ooOol05Goo00<007ooOol017oo0P00<7oo00<007oo
Ool08goo00L007ooOomoo`00Ool0000aOol00`00Oomoo`0QOol00`00Oomoo`02Ool00`00Ool0000`
Ool001Eoo`03001oogoo01Eoo`03001oogoo00=oo`03001oo`00031oo`03001oogoo02=oo`07001o
ogooOol007oo0000<Goo00<007ooOol08Goo00<007ooOol00Woo00<007oo0000<7oo000EOol00`00
Oomoo`0FOol00`00Oomoo`02Ool01000Oomoo`00;goo00<007ooOol097oo00H007ooOol007oo000`
Ool00`00Oomoo`0ROol00`00Oomoo`02Ool00`00Ool0000`Ool001Eoo`8001Moo`03001oogoo009o
o`04001oogoo000_Ool00`00Oomoo`0TOol01P00Oomoo`00Ool0031oo`03001oogoo029oo`03001o
ogoo009oo`03001oo`00031oo`005Goo00<007ooOol05goo00D007ooOomoo`0000=oo`03001oogoo
02aoo`03001oogoo02Aoo`03001oo`0000=oo`03001oogoo02eoo`03001oogoo029oo`03001oogoo
009oo`04001oogoo000_Ool001Eoo`03001oogoo01Moo`04001oogoo0004Ool00`00Oomoo`0/Ool0
0`00Oomoo`0TOol00`00Ool00003Ool00`00Oomoo`0]Ool00`00Oomoo`0ROol00`00Oomoo`02Ool0
1000Oomoo`00;goo000EOol00`00Oomoo`0HOol00`00Ool00005Ool00`00Oomoo`0ZOol00`00Oomo
o`0VOol20003Ool00`00Oomoo`0]Ool00`00Oomoo`0ROol01@00Oomoogoo00000goo00<007ooOol0
;Goo000EOol00`00Oomoo`0HOol00`00Ool00005Ool00`00Oomoo`0ZOol00`00Oomoo`0VOol00`00
Oomoo`03Ool00`00Oomoo`0/Ool00`00Oomoo`0ROol01@00Oomoogoo00000goo00<007ooOol0;Goo
000EOol00`00Oomoo`0IOol00`00Oomoo`04Ool00`00Oomoo`0ZOol00`00Oomoo`0VOol00`00Oomo
o`03Ool00`00Oomoo`0[Ool00`00Oomoo`0TOol01000Oomoo`0017oo00<007ooOol0;7oo000EOol2
000JOol00`00Oomoo`05Ool00`00Oomoo`0YOol00`00Oomoo`0/Ool00`00Oomoo`05Ool00`00Oomo
o`0SOol00`00Oomoo`0TOol01000Oomoo`0017oo00<007ooOol0;7oo000EOol00`00Oomoo`0QOol0
0`00Oomoo`0YOol00`00Oomoo`0]Ool00`00Oomoo`04Ool00`00Oomoo`0SOol00`00Oomoo`0TOol0
1000Oomoo`0017oo00<007ooOol01Goo00<007ooOol097oo000EOol00`00Oomoo`0QOol00`00Oomo
o`0YOol00`00Oomoo`0]Ool00`00Oomoo`03Ool00`00Ool0000TOol00`00Oomoo`0TOol00`00Ool0
0006Ool00`00Oomoo`03Ool2000VOol001Eoo`03001oogoo025oo`03001oogoo00=oo`03001oogoo
029oo`03001oogoo02moo`03001oogoo009oo`03001oo`0002Aoo`03001oogoo02Aoo`03001oo`00
00Ioo`03001oogoo00=oo`03001oo`0002Eoo`000Woo0P0017oo00<007ooOol00goo0P001Goo00<0
07ooOol08Woo00L007ooOomoo`00Ool0000SOol00`00Oomoo`0_Ool01@00Oomoogoo00000Woo00<0
07ooOol08Woo00<007ooOol097oo00<007oo00001goo00D007ooOomoo`00009oo`03001oogoo01Ao
o`D000Yoo`0000Eoo`00Oomoo`0000Qoo`04001oogoo0004Ool00`00Oomoo`0ROol01`00Oomoogoo
001oo`0002=oo`03001oogoo031oo`04001oogoo0003Ool00`00Oomoo`0POol00`00Oomoo`0UOol0
0`00Ool00007Ool01@00Oomoogoo00000Woo00<007ooOol057oo00<007ooOol037oo00001Goo001o
ogoo00002goo00<007ooOol00Woo0`008Woo00@007ooOol000=oo`03001oogoo021oo`03001oogoo
031oo`03001oo`0000Aoo`03001oogoo021oo`03001oogoo02Eoo`03001oo`0000Qoo`03001oo`00
00Aoo`03001oogoo01=oo`03001oogoo00aoo`0000Eoo`00Oomoo`0000Uoo`8000Eoo`03001oogoo
029oo`03001oo`0000Eoo`03001oogoo01moo`03001oogoo035oo`8000Aoo`03001oogoo021oo`03
001oogoo02Ioo`8000Qoo`03001oo`0000Aoo`03001oogoo01=oo`03001oogoo00aoo`0000Eoo`00
Oomoo`0000]oo`03001oogoo009oo`03001oogoo02=oo`8000Eoo`03001oogoo01ioo`03001oogoo
039oo`03001oogoo00Aoo`03001oogoo01moo`03001oogoo02Ioo`03001oogoo00Qoo`03001oogoo
00Aoo`03001oogoo019oo`03001oogoo00aoo`0000Eoo`00Oomoo`0000Qoo`04001oogoo0004Ool0
0`00Oomoo`0SOol00`00Oomoo`05Ool00`00Oomoo`0MOol00`00Oomoo`0iOol00`00Oomoo`0OOol0
0`00Oomoo`0VOol00`00Oomoo`0?Ool00`00Oomoo`0BOol00`00Oomoo`0<Ool0009oo`8000Yoo`80
00Eoo`03001oogoo02aoo`03001oogoo01aoo`03001oogoo03Yoo`03001oogoo01ioo`03001oogoo
02Ioo`03001oogoo011oo`03001oogoo011oo`03001oogoo00eoo`005Goo00<007ooOol0;7oo00<0
07ooOol077oo00<007ooOol0>goo00<007ooOol077oo00<007ooOol0>goo00<007ooOol03goo00<0
07ooOol03Goo000EOol2000^Ool00`00Oomoo`0KOol00`00Oomoo`0lOol00`00Oomoo`0KOol00`00
Oomoo`0kOol00`00Oomoo`0?Ool00`00Oomoo`0=Ool001Eoo`03001oogoo02ioo`03001oogoo01Yo
o`03001oogoo03eoo`03001oogoo01Yoo`03001oogoo03aoo`03001oogoo00ioo`03001oogoo00eo
o`005Goo00<007ooOol0;Woo00<007ooOol06Goo00<007ooOol0?Woo00<007ooOol06Woo00<007oo
Ool0?Goo00<007ooOol03Goo00<007ooOol03Goo000EOol00`00Oomoo`0_Ool00`00Oomoo`0HOol0
0`00Oomoo`0oOol00`00Oomoo`0IOol00`00Oomoo`0mOol00`00Oomoo`0=Ool00`00Oomoo`0=Ool0
01Eoo`03001oogoo02moo`03001oogoo01Qoo`03001oogoo03moo`03001oogoo01Uoo`03001oogoo
03ioo`03001oogoo00aoo`03001oogoo00eoo`005Goo00<007ooOol0<7oo00<007ooOol05goo00<0
07ooOol0?goo00<007ooOol067oo00<007ooOol0?goo00<007ooOol037oo00<007ooOol03Goo000E
Ool2000aOol00`00Oomoo`0GOol00`00Oomoo`10Ool00`00Oomoo`0GOol00`00Oomoo`0oOol00`00
Oomoo`0<Ool00`00Oomoo`0=Ool001Eoo`03001oogoo031oo`03001oogoo01Moo`03001oogoo041o
o`03001oogoo01Moo`03001oogoo041oo`03001oogoo00]oo`03001oogoo00eoo`005Goo00<007oo
Ool0<Goo00<007ooOol05Goo00<007ooOol0@Goo00<007ooOol05goo00<007ooOol0@7oo00<007oo
Ool02goo00<007ooOol03Goo000EOol00`00Oomoo`0aOol00`00Oomoo`0EOol00`00Oomoo`12Ool0
0`00Oomoo`0FOol00`00Oomoo`10Ool00`00Oomoo`0:Ool00`00Oomoo`0>Ool001Eoo`03001oogoo
035oo`03001oogoo01Eoo`03001oogoo049oo`03001oogoo01Eoo`03001oogoo049oo`03001oogoo
00Uoo`03001oogoo00ioo`005Goo0P00<Woo00<007ooOol05Goo00<007ooOol0@goo00<007ooOol0
2Woo00<007ooOol01goo00<007ooOol0@Woo00<007ooOol02Goo00<007ooOol03Woo000EOol00`00
Oomoo`0bOol00`00Oomoo`0;Ool00`00Oomoo`06Ool00`00Oomoo`13Ool00`00Oomoo`0:Ool00`00
Oomoo`07Ool00`00Oomoo`13Ool00`00Oomoo`08Ool00`00Oomoo`0>Ool001Eoo`03001oogoo039o
o`03001oogoo00]oo`03001oogoo00Ioo`03001oogoo04=oo`03001oogoo00Yoo`03001oogoo00Mo
o`03001oogoo04=oo`03001oogoo00Qoo`03001oogoo00ioo`005Goo00<007ooOol0<Woo00<007oo
Ool02goo00<007ooOol01Goo00<007ooOol0AGoo00<007ooOol02Goo0P0027oo00<007ooOol0@goo
00<007ooOol027oo00<007ooOol03Woo000EOol00`00Oomoo`0cOol00`00Oomoo`0:Ool00`00Oomo
o`05Ool00`00Oomoo`15Ool00`00Oomoo`09Ool20008Ool00`00Oomoo`14Ool00`00Oomoo`07Ool0
0`00Oomoo`0>Ool001Eoo`03001oogoo03=oo`03001oogoo00Yoo`8000Ioo`03001oogoo04Eoo`03
001oogoo00Qoo`03001oo`0000Moo`03001oogoo04Eoo`03001oogoo00Moo`03001oogoo00ioo`00
5Goo0P00=7oo00<007ooOol02Goo00<007oo00001Woo00<007ooOol0AGoo00<007ooOol027oo00<0
07oo00001goo00<007ooOol0AGoo00<007ooOol01goo00<007ooOol03Woo000EOol00`00Oomoo`0c
Ool00`00Oomoo`09Ool00`00Ool00006Ool00`00Oomoo`15Ool00`00Oomoo`08Ool00`00Ool00007
Ool00`00Oomoo`15Ool00`00Oomoo`07Ool00`00Oomoo`0>Ool001Eoo`03001oogoo03=oo`03001o
ogoo00Uoo`03001oo`0000Ioo`03001oogoo04Eoo`03001oogoo00Qoo`03001oo`0000Moo`03001o
ogoo04Eoo`03001oogoo00Moo`03001oogoo00ioo`005Goo00<007ooOol0<goo00<007ooOol02Goo
00<007oo00001Goo00<007ooOol0AWoo00<007ooOol027oo00@007ooOol000Ioo`03001oogoo04Eo
o`03001oogoo00Ioo`03001oogoo00moo`000Woo0P0017oo00<007ooOol017oo00<007ooOol00goo
00<007ooOol0<goo00<007ooOol02Goo00<007oo00001Goo00<007ooOol0Agoo00<007ooOol01goo
00@007ooOol000Eoo`03001oogoo04Ioo`03001oogoo00Ioo`03001oogoo00moo`0000Eoo`00Oomo
o`0000Yoo`03001oogoo00=oo`03001oogoo03=oo`03001oogoo00Uoo`03001oo`0000Eoo`03001o
ogoo04Moo`03001oogoo00Moo`04001oogoo0005Ool00`00Oomoo`17Ool00`00Oomoo`05Ool00`00
Oomoo`0?Ool00005Ool007ooOol00007Ool50004Ool3000dOol00`00Oomoo`08Ool00`00Ool00005
Ool00`00Oomoo`17Ool00`00Oomoo`07Ool01000Oomoo`001Goo00<007ooOol0Agoo00<007ooOol0
1Goo00<007ooOol03goo00001Goo001oogoo00001goo00@007ooOol000Eoo`03001oogoo03Aoo`03
001oogoo00Qoo`04001oogoo0004Ool00`00Oomoo`17Ool00`00Oomoo`07Ool01000Oomoo`001Goo
00<007ooOol0Agoo00<007ooOol01Goo00<007ooOol03goo00001Goo001oogoo000027oo00<007oo
00001Goo00<007ooOol0=7oo00<007ooOol01goo00D007ooOomoo`0000Aoo`03001oogoo04Moo`03
001oogoo00Ioo`05001oogooOol00005Ool00`00Oomoo`17Ool00`00Oomoo`05Ool00`00Oomoo`0?
Ool00005Ool007ooOol00009Ool20005Ool00`00Oomoo`0dOol00`00Oomoo`07Ool01@00Oomoogoo
000017oo00<007ooOol0Agoo00<007ooOol01Woo00<007ooOol00Woo00D007ooOomoo`0004Yoo`03
001oogoo00Eoo`03001oogoo00moo`000Woo0P002goo00<007ooOol00goo00<007ooOol0=7oo00<0
07ooOol01goo00D007ooOomoo`0000=oo`03001oogoo04Qoo`03001oogoo00Ioo`03001oogoo009o
o`05001oogooOol0001:Ool00`00Oomoo`05Ool00`00Oomoo`0?Ool001Eoo`03001oogoo03Aoo`03
001oogoo00Moo`05001oogooOol00003Ool00`00Oomoo`18Ool00`00Oomoo`06Ool00`00Oomoo`02
Ool01@00Oomoogoo0000BWoo00<007ooOol01Goo00<007ooOol03goo000EOol2000eOol00`00Oomo
o`07Ool01@00Oomoogoo00000goo00<007ooOol0B7oo00<007ooOol01Woo00<007ooOol00Woo00D0
07ooOomoo`0004Yoo`03001oogoo00Eoo`03001oogoo00moo`005Goo00<007ooOol0=7oo00<007oo
Ool01goo00D007ooOomoo`0000=oo`03001oogoo04Qoo`03001oogoo00Ioo`03001oogoo009oo`05
001oogooOol0001:Ool00`00Oomoo`05Ool00`00Oomoo`0?Ool001Eoo`03001oogoo03Aoo`03001o
ogoo00Moo`03001oogoo009oo`04001oogoo001:Ool00`00Oomoo`06Ool00`00Oomoo`03Ool01000
Oomoo`00BWoo00<007ooOol017oo00<007ooOol047oo000EOol00`00Oomoo`0dOol00`00Oomoo`07
Ool00`00Oomoo`02Ool01000Oomoo`00BWoo00<007ooOol01Woo00<007ooOol00goo00<007oo0000
C7oo00<007ooOol00goo00<007ooOol047oo000EOol00`00Oomoo`0dOol00`00Oomoo`06Ool00`00
Oomoo`03Ool00`00Ool0001<Ool00`00Oomoo`05Ool00`00Oomoo`03Ool00`00Ool0001<Ool00`00
Oomoo`03Ool00`00Oomoo`0@Ool001Eoo`8003Eoo`03001oogoo00Ioo`03001oogoo00=oo`03001o
o`0004aoo`03001oogoo00Aoo`03001oogoo00Aoo`03001oo`0004aoo`03001oogoo00=oo`03001o
ogoo011oo`005Goo00<007ooOol0=Goo00<007ooOol01Goo00<007ooOol00goo00<007oo0000C7oo
00<007ooOol017oo00<007ooOol017oo00<007oo0000C7oo00<007ooOol00goo00<007ooOol047oo
000EOol00`00Oomoo`0eOol00`00Oomoo`05Ool00`00Oomoo`03Ool00`00Ool0001<Ool00`00Oomo
o`04Ool00`00Oomoo`04Ool00`00Ool0001<Ool00`00Oomoo`03Ool00`00Oomoo`0@Ool001Eoo`03
001oogoo03Eoo`03001oogoo00Eoo`03001oogoo00=oo`03001oo`0004aoo`03001oogoo00Aoo`03
001oogoo00Eoo`03001oogoo04]oo`03001oogoo00=oo`03001oogoo011oo`005Goo00<007ooOol0
=Goo00<007ooOol01Goo00<007ooOol00goo00<007oo0000C7oo00<007ooOol017oo00<007ooOol0
1Goo00<007ooOol0Bgoo00<007ooOol00goo00<007ooOol047oo000EOol00`00Oomoo`0eOol00`00
Oomoo`05Ool00`00Oomoo`04Ool00`00Oomoo`1;Ool00`00Oomoo`04Ool00`00Oomoo`05Ool00`00
Oomoo`1;Ool00`00Oomoo`03Ool00`00Oomoo`0@Ool001Eoo`8003Ioo`03001oogoo00Eoo`03001o
ogoo00Aoo`03001oogoo04]oo`03001oogoo00Aoo`03001oogoo05=oo`03001oogoo00=oo`03001o
ogoo011oo`005Goo00<007ooOol0=Goo00<007ooOol017oo00<007ooOol01Goo00<007ooOol0Bgoo
00<007ooOol017oo00<007ooOol0Dgoo00<007ooOol00goo00<007ooOol047oo000EOol00`00Oomo
o`0eOol00`00Oomoo`04Ool00`00Oomoo`05Ool00`00Oomoo`1;Ool00`00Oomoo`04Ool00`00Oomo
o`1COol00`00Oomoo`02Ool00`00Oomoo`0AOol001Eoo`03001oogoo03Eoo`03001oogoo00Aoo`03
001oogoo05=oo`03001oogoo00=oo`03001oogoo05Eoo`05001oogooOol0000COol001Eoo`03001o
ogoo03Eoo`03001oogoo00Aoo`03001oogoo05=oo`03001oogoo00=oo`03001oogoo05Eoo`05001o
ogooOol0000COol001Eoo`03001oogoo03Eoo`03001oogoo00Aoo`03001oogoo05Aoo`03001oogoo
009oo`03001oogoo05Eoo`05001oogooOol0000COol001Eoo`8003Ioo`03001oogoo00Aoo`03001o
ogoo05Aoo`03001oogoo009oo`03001oogoo05Eoo`05001oogooOol0000COol001Eoo`03001oogoo
03Ioo`03001oogoo00=oo`03001oogoo05Aoo`03001oogoo009oo`03001oogoo05Eoo`05001oogoo
Ool0000COol001Eoo`03001oogoo03Ioo`03001oogoo00=oo`03001oogoo05Aoo`03001oogoo009o
o`03001oogoo05Eoo`05001oogooOol0000COol001Eoo`03001oogoo03Ioo`03001oogoo009oo`03
001oogoo05Eoo`03001oogoo009oo`03001oogoo05Eoo`05001oogooOol0000COol0009oo`8000Ao
o`03001oogoo009oo`<000Eoo`03001oogoo03Ioo`03001oogoo009oo`03001oogoo05Eoo`03001o
ogoo009oo`03001oogoo05Eoo`05001oogooOol0000COol00005Ool007ooOol0000;Ool00`00Oomo
o`02Ool00`00Oomoo`0fOol00`00Oomoo`02Ool00`00Oomoo`1EOol00`00Oomoo`02Ool00`00Oomo
o`1EOol01@00Oomoogoo00004goo00001Goo001oogoo00002goo00<007ooOol00Woo0`00=Woo00<0
07ooOol00Woo00<007ooOol0EGoo00D007ooOomoo`0005Qoo`05001oogooOol0000COol00005Ool0
07ooOol00008Ool30005Ool00`00Oomoo`0fOol00`00Oomoo`02Ool00`00Oomoo`1EOol01@00Oomo
ogoo0000F7oo00@007ooOol001Aoo`0000Eoo`00Oomoo`0000Qoo`03001oogoo00Eoo`03001oogoo
03Ioo`03001oogoo009oo`03001oogoo05Eoo`05001oogooOol0001IOol00`00Ool0000DOol00005
Ool007ooOol00008Ool00`00Oomoo`05Ool00`00Oomoo`0fOol00`00Oomoo`02Ool00`00Oomoo`1E
Ool01@00Oomoogoo0000FGoo00<007oo000057oo0002Ool20009Ool40004Ool00`00Oomoo`0fOol0
0`00Oomoo`02Ool00`00Oomoo`1EOol01@00Oomoogoo0000FGoo00<007oo000057oo000EOol00`00
Oomoo`0fOol01@00Oomoogoo0000FGoo00@007ooOol005Uoo`03001oo`0001Aoo`005Goo0P00=goo
00D007ooOomoo`0005Uoo`04001oogoo001IOol00`00Ool0000DOol001Eoo`03001oogoo03Moo`04
001oogoo001IOol01000Oomoo`00FGoo00<007oo000057oo000EOol00`00Oomoo`0gOol01000Oomo
o`00FGoo00@007ooOol005Uoo`03001oo`0001Aoo`005Goo00<007ooOol0=goo00@007ooOol005Uo
o`03001oo`0005Yoo`03001oo`0001Aoo`005Goo00<007ooOol0=goo00@007ooOol005Uoo`03001o
o`0005Yoo`03001oo`0001Aoo`005Goo0P00>7oo00@007ooOol005Uoo`03001oo`0005Yoo`03001o
o`0001Aoo`005Goo00<007ooOol0=goo00@007ooOol005Uoo`03001oo`0005]oo`03001oogoo01=o
o`005Goo00<007ooOol0=goo00<007oo0000FWoo00<007oo0000Fgoo00<007ooOol04goo000EOol0
0`00Oomoo`0gOol00`00Ool0001JOol00`00Ool0001KOol00`00Oomoo`0COol001Eoo`03001oogoo
03Moo`03001oo`0005Yoo`03001oo`0005]oo`03001oogoo01=oo`005Goo00<007ooOol0=goo00<0
07oo0000FWoo00<007oo0000Fgoo00<007ooOol04goo000EOol2000hOol00`00Ool0001KOol2001K
Ool00`00Oomoo`0COol001Eoo`03001oogoo03Moo`03001oo`0005]oo`03001oogoo071oo`005Goo
00<007ooOol0>7oo0P00Fgoo00<007ooOol0L7oo000EOol00`00Oomoo`0hOol2001KOol00`00Oomo
o`1`Ool001Eoo`03001oogoo03Qoo`03001oogoo05Yoo`03001oogoo071oo`005Goo00<007ooOol0
>7oo00<007ooOol0FWoo00<007ooOol0L7oo000EOol2000iOol00`00Oomoo`3=Ool001Eoo`03001o
ogoo03Qoo`03001oogoo0<eoo`005Goo00<007ooOol0>7oo00<007ooOol0cGoo000EOol00`00Oomo
o`3oOol9Ool0009oo`8000Aoo`03001oogoo00=oo`8000Eoo`03001oogoo0?moo`Uoo`0000Eoo`00
Oomoo`0000Qoo`04001oogoo0004Ool00`00Oomoo`3oOol9Ool00005Ool007ooOol00008Ool01000
Oomoo`0017oo0`00ogoo2Goo00001Goo001oogoo000027oo0`00ogoo4Goo00001Goo001oogoo0000
27oo00<007ooOol0ogoo4Goo00001Goo001oogoo000027oo00<007ooOol0ogoo4Goo0002Ool2000:
Ool3003oOol@Ool00?moob5oo`00ogoo8Goo003oOolQOol00001\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {505.25, 328.312}} -> {-7.41289, \
1.15065, 0.0770298, 0.00345893}}]
}, Open  ]],

Cell[TextData[{
  "This specifies that fixed step sizes should be used for the method ",
  StyleBox["ExplicitRungeKutta", "MR"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(sol\  = \ 
        NDSolve[system, StartingStepSize \[Rule] 1/10, \ 
          Method \[Rule] {"\<FixedStep\>", \ 
              Method \[Rule] "\<ExplicitRungeKutta\>"}];\)\), "\
\[IndentingNewLine]", 
    \(\(StepDataPlot[sol, PlotRange \[Rule] {0, \ 0.2}];\)\)}], "Input",
  CellLabel->"In[51]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0478583 1.61546e-17 3.09017 [
[.2631 -0.0125 -3 -9 ]
[.2631 -0.0125 3 0 ]
[.50239 -0.0125 -6 -9 ]
[.50239 -0.0125 6 0 ]
[.74168 -0.0125 -6 -9 ]
[.74168 -0.0125 6 0 ]
[.98098 -0.0125 -6 -9 ]
[.98098 -0.0125 6 0 ]
[.01131 .15451 -24 -4.5 ]
[.01131 .15451 0 4.5 ]
[.01131 .30902 -18 -4.5 ]
[.01131 .30902 0 4.5 ]
[.01131 .46353 -24 -4.5 ]
[.01131 .46353 0 4.5 ]
[.01131 .61803 -18 -4.5 ]
[.01131 .61803 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.2631 0 m
.2631 .00625 L
s
[(5)] .2631 -0.0125 0 1 Mshowa
.50239 0 m
.50239 .00625 L
s
[(10)] .50239 -0.0125 0 1 Mshowa
.74168 0 m
.74168 .00625 L
s
[(15)] .74168 -0.0125 0 1 Mshowa
.98098 0 m
.98098 .00625 L
s
[(20)] .98098 -0.0125 0 1 Mshowa
.125 Mabswid
.07167 0 m
.07167 .00375 L
s
.11953 0 m
.11953 .00375 L
s
.16738 0 m
.16738 .00375 L
s
.21524 0 m
.21524 .00375 L
s
.31096 0 m
.31096 .00375 L
s
.35882 0 m
.35882 .00375 L
s
.40668 0 m
.40668 .00375 L
s
.45453 0 m
.45453 .00375 L
s
.55025 0 m
.55025 .00375 L
s
.59811 0 m
.59811 .00375 L
s
.64597 0 m
.64597 .00375 L
s
.69383 0 m
.69383 .00375 L
s
.78954 0 m
.78954 .00375 L
s
.8374 0 m
.8374 .00375 L
s
.88526 0 m
.88526 .00375 L
s
.93312 0 m
.93312 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
.02381 .15451 m
.03006 .15451 L
s
[(0.05)] .01131 .15451 1 0 Mshowa
.02381 .30902 m
.03006 .30902 L
s
[(0.1)] .01131 .30902 1 0 Mshowa
.02381 .46353 m
.03006 .46353 L
s
[(0.15)] .01131 .46353 1 0 Mshowa
.02381 .61803 m
.03006 .61803 L
s
[(0.2)] .01131 .61803 1 0 Mshowa
.125 Mabswid
.02381 .0309 m
.02756 .0309 L
s
.02381 .0618 m
.02756 .0618 L
s
.02381 .09271 m
.02756 .09271 L
s
.02381 .12361 m
.02756 .12361 L
s
.02381 .18541 m
.02756 .18541 L
s
.02381 .21631 m
.02756 .21631 L
s
.02381 .24721 m
.02756 .24721 L
s
.02381 .27812 m
.02756 .27812 L
s
.02381 .33992 m
.02756 .33992 L
s
.02381 .37082 m
.02756 .37082 L
s
.02381 .40172 m
.02756 .40172 L
s
.02381 .43262 m
.02756 .43262 L
s
.02381 .49443 m
.02756 .49443 L
s
.02381 .52533 m
.02756 .52533 L
s
.02381 .55623 m
.02756 .55623 L
s
.02381 .58713 m
.02756 .58713 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .30902 m
.0286 .30902 L
.03338 .30902 L
.03817 .30902 L
.04295 .30902 L
.04774 .30902 L
.05252 .30902 L
.05731 .30902 L
.0621 .30902 L
.06688 .30902 L
.07167 .30902 L
.07645 .30902 L
.08124 .30902 L
.08603 .30902 L
.09081 .30902 L
.0956 .30902 L
.10038 .30902 L
.10517 .30902 L
.10995 .30902 L
.11474 .30902 L
.11953 .30902 L
.12431 .30902 L
.1291 .30902 L
.13388 .30902 L
.13867 .30902 L
.14346 .30902 L
.14824 .30902 L
.15303 .30902 L
.15781 .30902 L
.1626 .30902 L
.16738 .30902 L
.17217 .30902 L
.17696 .30902 L
.18174 .30902 L
.18653 .30902 L
.19131 .30902 L
.1961 .30902 L
.20089 .30902 L
.20567 .30902 L
.21046 .30902 L
.21524 .30902 L
.22003 .30902 L
.22481 .30902 L
.2296 .30902 L
.23439 .30902 L
.23917 .30902 L
.24396 .30902 L
.24874 .30902 L
.25353 .30902 L
.25832 .30902 L
Mistroke
.2631 .30902 L
.26789 .30902 L
.27267 .30902 L
.27746 .30902 L
.28224 .30902 L
.28703 .30902 L
.29182 .30902 L
.2966 .30902 L
.30139 .30902 L
.30617 .30902 L
.31096 .30902 L
.31575 .30902 L
.32053 .30902 L
.32532 .30902 L
.3301 .30902 L
.33489 .30902 L
.33967 .30902 L
.34446 .30902 L
.34925 .30902 L
.35403 .30902 L
.35882 .30902 L
.3636 .30902 L
.36839 .30902 L
.37318 .30902 L
.37796 .30902 L
.38275 .30902 L
.38753 .30902 L
.39232 .30902 L
.3971 .30902 L
.40189 .30902 L
.40668 .30902 L
.41146 .30902 L
.41625 .30902 L
.42103 .30902 L
.42582 .30902 L
.43061 .30902 L
.43539 .30902 L
.44018 .30902 L
.44496 .30902 L
.44975 .30902 L
.45453 .30902 L
.45932 .30902 L
.46411 .30902 L
.46889 .30902 L
.47368 .30902 L
.47846 .30902 L
.48325 .30902 L
.48804 .30902 L
.49282 .30902 L
.49761 .30902 L
Mistroke
.50239 .30902 L
.50718 .30902 L
.51196 .30902 L
.51675 .30902 L
.52154 .30902 L
.52632 .30902 L
.53111 .30902 L
.53589 .30902 L
.54068 .30902 L
.54547 .30902 L
.55025 .30902 L
.55504 .30902 L
.55982 .30902 L
.56461 .30902 L
.56939 .30902 L
.57418 .30902 L
.57897 .30902 L
.58375 .30902 L
.58854 .30902 L
.59332 .30902 L
.59811 .30902 L
.6029 .30902 L
.60768 .30902 L
.61247 .30902 L
.61725 .30902 L
.62204 .30902 L
.62682 .30902 L
.63161 .30902 L
.6364 .30902 L
.64118 .30902 L
.64597 .30902 L
.65075 .30902 L
.65554 .30902 L
.66033 .30902 L
.66511 .30902 L
.6699 .30902 L
.67468 .30902 L
.67947 .30902 L
.68425 .30902 L
.68904 .30902 L
.69383 .30902 L
.69861 .30902 L
.7034 .30902 L
.70818 .30902 L
.71297 .30902 L
.71776 .30902 L
.72254 .30902 L
.72733 .30902 L
.73211 .30902 L
.7369 .30902 L
Mistroke
.74168 .30902 L
.74647 .30902 L
.75126 .30902 L
.75604 .30902 L
.76083 .30902 L
.76561 .30902 L
.7704 .30902 L
.77519 .30902 L
.77997 .30902 L
.78476 .30902 L
.78954 .30902 L
.79433 .30902 L
.79911 .30902 L
.8039 .30902 L
.80869 .30902 L
.81347 .30902 L
.81826 .30902 L
.82304 .30902 L
.82783 .30902 L
.83262 .30902 L
.8374 .30902 L
.84219 .30902 L
.84697 .30902 L
.85176 .30902 L
.85654 .30902 L
.86133 .30902 L
.86612 .30902 L
.8709 .30902 L
.87569 .30902 L
.88047 .30902 L
.88526 .30902 L
.89005 .30902 L
.89483 .30902 L
.89962 .30902 L
.9044 .30902 L
.90919 .30902 L
.91397 .30902 L
.91876 .30902 L
.92355 .30902 L
.92833 .30902 L
.93312 .30902 L
.9379 .30902 L
.94269 .30902 L
.94748 .30902 L
.95226 .30902 L
.95705 .30902 L
.96183 .30902 L
.96662 .30902 L
.9714 .30902 L
.97619 .30902 L
Mistroke
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[51]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool005Yoo`<003Qoo`@000=oo`8003Eoo`@0009oo`<003Eoo`@000=oo`8000Ioo`00GGoo00<007oo
Ool0=goo00D007ooOomoo`00009oo`03001oogoo03Aoo`03001oogoo00Aoo`03001oogoo039oo`03
001oogoo00=oo`04001oogoo0005Ool005eoo`03001oogoo03Moo`05001oogooOol00002Ool00`00
Oomoo`0dOol00`00Oomoo`04Ool00`00Oomoo`0cOol00`00Oomoo`02Ool01000Oomoo`001Goo001J
Ool3000jOol01@00Oomoogoo00000Woo00<007ooOol0=7oo00@007ooOomoo`<003Moo`05001oogoo
Ool00002Ool00`00Oomoo`03Ool005Yoo`03001oogoo03Yoo`05001oogooOol00002Ool00`00Oomo
o`0dOol01@00Oomoogoo0000>Woo00@007ooOol0009oo`03001oogoo00=oo`00FWoo00<007ooOol0
>7oo0`000goo00@007ooOol003Aoo`<000=oo`03001oogoo03Eoo`04001oogoo0002Ool01000Oomo
o`001Goo001JOol4000iOol00`00Oomoo`02Ool2000gOol01000Oomoogoo1000=Goo0P0017oo0P00
1Woo003oOolQOol00?moob5oo`00ogoo8Goo000GOooo00060004Ool001ioo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03
001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo00Moo`00
7Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007oo
Ool0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol0
0`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3o
Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`80
0?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`00
7Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007oo
Ool0ogoo000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol0
0`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3o
Ool001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001o
ogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`00
17oo0P0017oo00<007ooOol00goo0P000goo0`001Woo00<007ooOol0ogoo0003Ool01000Oomoo`00
27oo00@007ooOol000Eoo`03001oogoo00=oo`03001oogoo0?moo`000goo00@007ooOol000Qoo`04
001oogoo0005Ool00`00Oomoo`03Ool2003oOol1Ool000=oo`04001oogoo0008Ool01000Oomoo`00
0Woo0`001Woo00<007ooOol0ogoo0003Ool01000Oomoo`0027oo00@007ooOol0009oo`03001oogoo
00Ioo`03001oogoo0?moo`000goo00@007ooOol000Qoo`04001oogoo0002Ool00`00Oomoo`06Ool0
0`00Oomoo`3oOol000Aoo`8000Yoo`8000=oo`@000Eoo`03001oogoo0?moo`007Woo00<007ooOol0
ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`800?mo
o`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo
00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0
ogoo000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001oogoo
0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`002Woo
0P0017oo00<007ooOol00Woo10001Goo00<007ooOol0ogoo0009Ool01000Oomoo`002Woo00<007oo
Ool017oo00<007ooOol0ogoo0009Ool01000Oomoo`002Woo00<007ooOol017oom`002goo0009Ool0
1000Oomoo`002Woo00<007ooOol017oo00<007ooOol0ogoo0009Ool01000Oomoo`002Woo00<007oo
Ool017oo00<007ooOol0ogoo0009Ool01000Oomoo`0027oo0`001Woo00<007ooOol0ogoo000:Ool2
000;Ool00`00Oomoo`04Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0
ogoo000NOol00`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol2003oOol1Ool001ioo`03001oogoo
0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo
00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo
000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo
0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol000Aoo`8000Aoo`03001oogoo009o
o`@0009oo`<000Ioo`03001oogoo0?moo`000goo00@007ooOol000Yoo`03001oogoo00Aoo`03001o
ogoo00=oo`03001oogoo0?moo`000goo00@007ooOol000Yoo`03001oogoo00Aoo`03001oogoo00=o
o`800?moo`5oo`000goo00@007ooOol000Yoo`04001oogooOol30006Ool00`00Oomoo`3oOol000=o
o`04001oogoo000:Ool01@00Oomoogoo000027oo00<007ooOol0ogoo0003Ool01000Oomoo`0027oo
0`000goo00<007ooOol01Woo00<007ooOol0ogoo0004Ool2000;Ool01000Oomoogoo10001Goo00<0
07ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo
000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomo
o`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001io
o`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?mo
o`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<0
07ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000N
Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomo
o`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol2003oOol1Ool001ioo`03
001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?mo
o`007Woo00<007ooOol0ogoo000:Ool20004Ool00`00Oomoo`02Ool40005Ool00`00Oomoo`3oOol0
00Uoo`04001oogoo0008Ool00`00Oomoo`06Ool00`00Oomoo`3oOol000Uoo`04001oogoo0009Ool0
0`00Oomoo`05Ool2003oOol1Ool000Uoo`04001oogoo000:Ool00`00Oomoo`3oOol7Ool000Uoo`04
001oogoo000;Ool00`00Oomoo`3oOol6Ool000Uoo`04001oogoo0008Ool01000Oomoo`00ogoo27oo
000:Ool2000:Ool2003oOol9Ool00001\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {276.312, 99.375}} -> {-8.45975, \
0.108784, 0.0806123, 0.00124846}}]
}, Open  ]],

Cell[TextData[{
  "The option ",
  StyleBox["MaxStepFraction", "MR"],
  " provides an absolute bound on the step size that depends on the \
integration interval."
}], "Text"],

Cell[TextData[{
  "Since the default value of ",
  StyleBox["MaxStepFraction", "MR"],
  " is ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(1/10\)\(,\)\)\)]],
  " the step size in this example is bounded by one-tenth of the integration \
interval, which leads to using a constant step size of ",
  Cell[BoxData[
      \(TraditionalForm\`1/20\)]],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(time\  = \ {T, \ 0, \ 1/2};\)\), "\[IndentingNewLine]", 
    \(\(sol\  = \ 
        NDSolve[system, time, \ StartingStepSize \[Rule] 1/10, \ 
          Method \[Rule] {"\<FixedStep\>", \ 
              Method \[Rule] "\<ExplicitRungeKutta\>"}];\)\), "\
\[IndentingNewLine]", 
    \(\(StepDataPlot[sol, PlotRange \[Rule] {0, \ 0.2}];\)\)}], "Input",
  CellLabel->"In[48]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 2.1164 1.61546e-17 3.09017 [
[.23545 -0.0125 -9 -9 ]
[.23545 -0.0125 9 0 ]
[.44709 -0.0125 -9 -9 ]
[.44709 -0.0125 9 0 ]
[.65873 -0.0125 -9 -9 ]
[.65873 -0.0125 9 0 ]
[.87037 -0.0125 -9 -9 ]
[.87037 -0.0125 9 0 ]
[.01131 .15451 -24 -4.5 ]
[.01131 .15451 0 4.5 ]
[.01131 .30902 -18 -4.5 ]
[.01131 .30902 0 4.5 ]
[.01131 .46353 -24 -4.5 ]
[.01131 .46353 0 4.5 ]
[.01131 .61803 -18 -4.5 ]
[.01131 .61803 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.23545 0 m
.23545 .00625 L
s
[(0.1)] .23545 -0.0125 0 1 Mshowa
.44709 0 m
.44709 .00625 L
s
[(0.2)] .44709 -0.0125 0 1 Mshowa
.65873 0 m
.65873 .00625 L
s
[(0.3)] .65873 -0.0125 0 1 Mshowa
.87037 0 m
.87037 .00625 L
s
[(0.4)] .87037 -0.0125 0 1 Mshowa
.125 Mabswid
.06614 0 m
.06614 .00375 L
s
.10847 0 m
.10847 .00375 L
s
.15079 0 m
.15079 .00375 L
s
.19312 0 m
.19312 .00375 L
s
.27778 0 m
.27778 .00375 L
s
.32011 0 m
.32011 .00375 L
s
.36243 0 m
.36243 .00375 L
s
.40476 0 m
.40476 .00375 L
s
.48942 0 m
.48942 .00375 L
s
.53175 0 m
.53175 .00375 L
s
.57407 0 m
.57407 .00375 L
s
.6164 0 m
.6164 .00375 L
s
.70106 0 m
.70106 .00375 L
s
.74339 0 m
.74339 .00375 L
s
.78571 0 m
.78571 .00375 L
s
.82804 0 m
.82804 .00375 L
s
.9127 0 m
.9127 .00375 L
s
.95503 0 m
.95503 .00375 L
s
.99735 0 m
.99735 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
.02381 .15451 m
.03006 .15451 L
s
[(0.05)] .01131 .15451 1 0 Mshowa
.02381 .30902 m
.03006 .30902 L
s
[(0.1)] .01131 .30902 1 0 Mshowa
.02381 .46353 m
.03006 .46353 L
s
[(0.15)] .01131 .46353 1 0 Mshowa
.02381 .61803 m
.03006 .61803 L
s
[(0.2)] .01131 .61803 1 0 Mshowa
.125 Mabswid
.02381 .0309 m
.02756 .0309 L
s
.02381 .0618 m
.02756 .0618 L
s
.02381 .09271 m
.02756 .09271 L
s
.02381 .12361 m
.02756 .12361 L
s
.02381 .18541 m
.02756 .18541 L
s
.02381 .21631 m
.02756 .21631 L
s
.02381 .24721 m
.02756 .24721 L
s
.02381 .27812 m
.02756 .27812 L
s
.02381 .33992 m
.02756 .33992 L
s
.02381 .37082 m
.02756 .37082 L
s
.02381 .40172 m
.02756 .40172 L
s
.02381 .43262 m
.02756 .43262 L
s
.02381 .49443 m
.02756 .49443 L
s
.02381 .52533 m
.02756 .52533 L
s
.02381 .55623 m
.02756 .55623 L
s
.02381 .58713 m
.02756 .58713 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .15451 m
.12963 .15451 L
.23545 .15451 L
.34127 .15451 L
.44709 .15451 L
.55291 .15451 L
.65873 .15451 L
.76455 .15451 L
.87037 .15451 L
.97619 .15451 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[48]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool004ioo`8000Aoo`03001oogoo009oo`@002Qoo`8000Aoo`03001oogoo009oo`@002Qoo`8000Ao
o`03001oogoo00=oo`8002Uoo`8000Aoo`03001oogoo00Aoo`03001oogoo01eoo`00CGoo00@007oo
Ool000Yoo`03001oogoo02Ioo`04001oogoo0008Ool00`00Oomoo`0XOol01000Oomoo`0027oo00@0
07ooOol002Moo`04001oogoo000:Ool00`00Oomoo`0MOol004eoo`04001oogoo000:Ool00`00Oomo
o`0VOol01000Oomoo`002Goo00<007ooOol09goo00@007ooOol000]oo`03001oogoo02Eoo`04001o
ogoo0007Ool5000NOol004eoo`04001oogoo000:Ool00`00Oomoo`0VOol01000Oomoo`002Woo00<0
07ooOol09Woo00@007ooOol000Uoo`8002Qoo`04001oogoo0007Ool01000Oomoo`007goo001=Ool0
1000Oomoo`002Woo00<007ooOol09Woo00@007ooOol000]oo`03001oogoo02Eoo`04001oogoo000;
Ool00`00Oomoo`0UOol01000Oomoo`0027oo00<007oo00007goo001=Ool01000Oomoo`0027oo0`00
:7oo00@007ooOol000Qoo`04001oogoo000WOol01000Oomoo`0027oo00@007ooOol002Moo`04001o
ogoo0009Ool2000OOol004ioo`8000]oo`03001oogoo02Moo`8000Yoo`8002Uoo`8000Yoo`8002Uo
o`8000]oo`03001oogoo01eoo`00ogoo8Goo003oOolQOol00?moob5oo`0067ooo`001@0017oo000N
Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08
Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08
Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08
Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08
Ool00`00Oomoo`08Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`03Ool001ioo`03001o
ogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`00
7Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0
ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol0
0`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol2003oOol1Ool0
01ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001o
ogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`00
7Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0
ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol0
0`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol000Eoo`8000Ao
o`03001oogoo00=oo`8000=oo`<000Eoo`03001oogoo0?moo`0017oo00@007ooOol000Qoo`04001o
ogoo0005Ool00`00Oomoo`02Ool00`00Oomoo`3oOol000Aoo`04001oogoo0008Ool01000Oomoo`00
1Goo00<007ooOol00Woon0002Woo0004Ool01000Oomoo`0027oo00@007ooOol0009oo`<000Eoo`03
001oogoo0?moo`0017oo00@007ooOol000Qoo`04001oogoo0002Ool00`00Oomoo`05Ool00`00Oomo
o`3oOol000Aoo`04001oogoo0008Ool01000Oomoo`000Woo00<007ooOol01Goo00<007ooOol0ogoo
0005Ool2000:Ool20003Ool40004Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<0
07ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000N
Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomo
o`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol2003oOol1Ool001ioo`03
001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?mo
o`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00
ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000N
Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomo
o`3oOol001ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03
001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol000]oo`8000Aoo`03001o
ogoo009oo`@000Aoo`03001oogoo0?moo`002Woo00@007ooOol000Yoo`03001oogoo00=oo`03001o
ogoo0?moo`002Woo00@007ooOol000Yoo`03001oogoo00=oo`<00?moo`002Woo00@007ooOol000Yo
o`03001oogoo00=oo`03001oogoo0?moo`002Woo00@007ooOol000Yoo`03001oogoo00=oo`03001o
ogoo0?moo`002Woo00@007ooOol000Qoo`<000Eoo`03001oogoo0?moo`002goo0P002goo00<007oo
Ool00goo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<0
07ooOol0ogoo000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000N
Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomo
o`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001ioo`03
001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?mo
o`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`800?moo`5oo`007Woo00<007oo
Ool0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000N
Ool00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol2003oOol1
Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03
001oogoo0?moo`007Woo00<007ooOol0ogoo0005Ool20004Ool00`00Oomoo`02Ool40002Ool30005
Ool00`00Oomoo`3oOol000Aoo`04001oogoo000:Ool00`00Oomoo`04Ool00`00Oomoo`02Ool00`00
Oomoo`3oOol000Aoo`04001oogoo000:Ool00`00Oomoo`04Ool00`00Oomoo`02Ool3003oOol000Ao
o`04001oogoo000:Ool01000Oomoogoo0`001Goo00<007ooOol0ogoo0004Ool01000Oomoo`002Woo
00D007ooOomoo`0000Moo`03001oogoo0?moo`0017oo00@007ooOol000Qoo`<000=oo`03001oogoo
00Eoo`03001oogoo0?moo`001Goo0P002goo00@007ooOomoo`@000Aoo`03001oogoo0?moo`007Woo
00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo
000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol0
01ioo`800?moo`5oo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo
0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo
00<007ooOol0ogoo000NOol2003oOol1Ool001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo
000NOol00`00Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00
Oomoo`3oOol001ioo`03001oogoo0?moo`007Woo0P00ogoo0Goo000NOol00`00Oomoo`3oOol001io
o`03001oogoo0?moo`007Woo00<007ooOol0ogoo000NOol00`00Oomoo`3oOol001ioo`03001oogoo
0?moo`002goo0P0017oo00<007ooOol00Woo100017oo00<007ooOol0ogoo000:Ool01000Oomoo`00
27oo00<007ooOol01Goo00<007ooOol0ogoo000:Ool01000Oomoo`002Goo00<007ooOol017oo0`00
ogoo000:Ool01000Oomoo`002Woo00<007ooOol0ogoo1Woo000:Ool01000Oomoo`002goo00<007oo
Ool0ogoo1Goo000:Ool01000Oomoo`0027oo00@007ooOol00?moo`Moo`002goo0P002Woo0P00ogoo
27oo0000\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {573, 396.062}} -> {-0.192276, \
0.479187, 0.00182289, 0.00124846}}]
}, Open  ]],

Cell[TextData[{
  "By setting the value of ",
  StyleBox["MaxStepFraction", "MR"],
  " to a different value, the dependence of the step size on the integration \
interval can be relaxed or removed entirely."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(sol\  = \ 
        NDSolve[system, time, \ StartingStepSize \[Rule] 1/10, 
          MaxStepFraction \[Rule] 
            Infinity, \ \ Method \[Rule] {"\<FixedStep\>", \ 
              Method \[Rule] "\<ExplicitRungeKutta\>"}];\)\), "\
\[IndentingNewLine]", 
    \(\(StepDataPlot[sol, PlotRange \[Rule] {0, \ 0.2}];\)\)}], "Input",
  CellLabel->"In[53]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 2.38095 1.61546e-17 3.09017 [
[.2619 -0.0125 -9 -9 ]
[.2619 -0.0125 9 0 ]
[.5 -0.0125 -9 -9 ]
[.5 -0.0125 9 0 ]
[.7381 -0.0125 -9 -9 ]
[.7381 -0.0125 9 0 ]
[.97619 -0.0125 -9 -9 ]
[.97619 -0.0125 9 0 ]
[.01131 .15451 -24 -4.5 ]
[.01131 .15451 0 4.5 ]
[.01131 .30902 -18 -4.5 ]
[.01131 .30902 0 4.5 ]
[.01131 .46353 -24 -4.5 ]
[.01131 .46353 0 4.5 ]
[.01131 .61803 -18 -4.5 ]
[.01131 .61803 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.2619 0 m
.2619 .00625 L
s
[(0.1)] .2619 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(0.2)] .5 -0.0125 0 1 Mshowa
.7381 0 m
.7381 .00625 L
s
[(0.3)] .7381 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(0.4)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07143 0 m
.07143 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.16667 0 m
.16667 .00375 L
s
.21429 0 m
.21429 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.35714 0 m
.35714 .00375 L
s
.40476 0 m
.40476 .00375 L
s
.45238 0 m
.45238 .00375 L
s
.54762 0 m
.54762 .00375 L
s
.59524 0 m
.59524 .00375 L
s
.64286 0 m
.64286 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.78571 0 m
.78571 .00375 L
s
.83333 0 m
.83333 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
.02381 .15451 m
.03006 .15451 L
s
[(0.05)] .01131 .15451 1 0 Mshowa
.02381 .30902 m
.03006 .30902 L
s
[(0.1)] .01131 .30902 1 0 Mshowa
.02381 .46353 m
.03006 .46353 L
s
[(0.15)] .01131 .46353 1 0 Mshowa
.02381 .61803 m
.03006 .61803 L
s
[(0.2)] .01131 .61803 1 0 Mshowa
.125 Mabswid
.02381 .0309 m
.02756 .0309 L
s
.02381 .0618 m
.02756 .0618 L
s
.02381 .09271 m
.02756 .09271 L
s
.02381 .12361 m
.02756 .12361 L
s
.02381 .18541 m
.02756 .18541 L
s
.02381 .21631 m
.02756 .21631 L
s
.02381 .24721 m
.02756 .24721 L
s
.02381 .27812 m
.02756 .27812 L
s
.02381 .33992 m
.02756 .33992 L
s
.02381 .37082 m
.02756 .37082 L
s
.02381 .40172 m
.02756 .40172 L
s
.02381 .43262 m
.02756 .43262 L
s
.02381 .49443 m
.02756 .49443 L
s
.02381 .52533 m
.02756 .52533 L
s
.02381 .55623 m
.02756 .55623 L
s
.02381 .58713 m
.02756 .58713 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .30902 m
.2619 .30902 L
.5 .30902 L
.7381 .30902 L
.97619 .30902 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[53]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool005=oo`8000Aoo`03001oogoo009oo`@002moo`8000Aoo`03001oogoo009oo`@002moo`8000Ao
o`03001oogoo00=oo`80031oo`8000Aoo`03001oogoo00Aoo`03001oogoo00=oo`00DWoo00@007oo
Ool000Yoo`03001oogoo02eoo`04001oogoo0008Ool00`00Oomoo`0_Ool01000Oomoo`0027oo00@0
07ooOol002ioo`04001oogoo000:Ool00`00Oomoo`03Ool0059oo`04001oogoo000:Ool00`00Oomo
o`0]Ool01000Oomoo`002Goo00<007ooOol0;Woo00@007ooOol000]oo`03001oogoo02aoo`04001o
ogoo0007Ool50004Ool0059oo`04001oogoo000:Ool00`00Oomoo`0]Ool01000Oomoo`002Woo00<0
07ooOol0;Goo00@007ooOol000Uoo`8002moo`04001oogoo0007Ool01000Oomoo`001Goo001BOol0
1000Oomoo`002Woo00<007ooOol0;Goo00@007ooOol000]oo`03001oogoo02aoo`04001oogoo000;
Ool00`00Oomoo`0/Ool01000Oomoo`0027oo00<007oo00001Goo001BOol01000Oomoo`0027oo0`00
;goo00@007ooOol000Qoo`04001oogoo000^Ool01000Oomoo`0027oo00@007ooOol002ioo`04001o
ogoo0009Ool20005Ool005=oo`8000]oo`03001oogoo02ioo`8000Yoo`80031oo`8000Yoo`80031o
o`8000]oo`03001oogoo00=oo`00ogoo8Goo003oOolQOol00?moob5oo`005gooo`001@001Goo000M
Ool00`00Oomoo`09Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool00`00Oomoo`09Ool00`00Oomoo`09
Ool00`00Oomoo`0:Ool00`00Oomoo`09Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool00`00Oomoo`09
Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool00`00Oomoo`09Ool00`00Oomoo`09Ool00`00Oomoo`0:
Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool00`00Oomoo`09Ool00`00Oomoo`09Ool00`00Oomoo`0:
Ool00`00Oomoo`09Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol0
0`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol0
0`00Oomoo`3oOol1Ool001eoo`800?moo`9oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomo
o`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomo
o`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol2003oOol2
Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1
Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1
Ool001eoo`03001oogoo0?moo`5oo`007Goo0P00ogoo0Woo000MOol00`00Oomoo`3oOol1Ool001eo
o`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eo
o`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eo
o`800?moo`9oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001o
ogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool000=oo`8000Ao
o`03001oogoo00=oo`8000=oo`<000Ioo`03001oogoo0?moo`5oo`000Woo00@007ooOol000Qoo`04
001oogoo0005Ool00`00Oomoo`03Ool00`00Oomoo`3oOol1Ool0009oo`04001oogoo0008Ool01000
Oomoo`001Goo00<007ooOol00goo0P00ogoo0Woo0002Ool01000Oomoo`0027oo00@007ooOol0009o
o`<000Ioo`03001oogoo0?moo`5oo`000Woo00@007ooOol000Qoo`04001oogoo0002Ool00`00Oomo
o`06Ool00`00Oomoo`3oOol1Ool0009oo`04001oogoo0008Ool01000Oomoo`000Woo00<007ooOol0
1Woo00<007ooOol0ogoo0Goo0003Ool2000:Ool20003Ool40005Ool00`00Oomoo`3oOol1Ool001eo
o`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eo
o`800?moo`9oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001o
ogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001o
ogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol2003oOol2Ool001eoo`03001oogoo0?mo
o`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?mo
o`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?mo
o`5oo`007Goo0P00ogoo0Woo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`00
7Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`00
7Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`800?moo`9oo`007Goo00<0
07ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<0
07ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool000Uoo`8000Aoo`03001oogoo009oo`@000Eo
o`03001oogoo0?moo`5oo`0027oo00@007ooOol000Yoo`03001oogoo00Aoo`03001oogoo0?moo`5o
o`0027oo00@007ooOol000Yoo`03001oogoo00AoooP000]oo`0027oo00@007ooOol000Yoo`03001o
ogoo00Aoo`03001oogoo0?moo`5oo`0027oo00@007ooOol000Yoo`03001oogoo00Aoo`03001oogoo
0?moo`5oo`0027oo00@007ooOol000Qoo`<000Ioo`03001oogoo0?moo`5oo`002Goo0P002goo00<0
07ooOol017oo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?mo
o`5oo`007Goo00<007ooOol0ogoo0Goo000MOol2003oOol2Ool001eoo`03001oogoo0?moo`5oo`00
7Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`00
7Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`00
7Goo0P00ogoo0Woo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<0
07ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<0
07ooOol0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`800?moo`9oo`007Goo00<007ooOol0
ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0
ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0
ogoo0Goo000MOol2003oOol2Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo
000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo
0003Ool20004Ool00`00Oomoo`02Ool40002Ool30006Ool00`00Oomoo`3oOol1Ool0009oo`04001o
ogoo000:Ool00`00Oomoo`04Ool00`00Oomoo`03Ool00`00Oomoo`3oOol1Ool0009oo`04001oogoo
000:Ool00`00Oomoo`04Ool00`00Oomoo`03Ool2003oOol2Ool0009oo`04001oogoo000:Ool01000
Oomoogoo0`001Woo00<007ooOol0ogoo0Goo0002Ool01000Oomoo`002Woo00D007ooOomoo`0000Qo
o`03001oogoo0?moo`5oo`000Woo00@007ooOol000Qoo`<000=oo`03001oogoo00Ioo`03001oogoo
0?moo`5oo`000goo0P002goo00@007ooOomoo`@000Eoo`03001oogoo0?moo`5oo`007Goo00<007oo
Ool0ogoo0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo0P00ogoo
0Woo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo
0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo
0Goo000MOol00`00Oomoo`3oOol1Ool001eoo`800?moo`9oo`007Goo00<007ooOol0ogoo0Goo000M
Ool00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000M
Ool00`00Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000M
Ool2003oOol2Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00
Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00
Oomoo`3oOol1Ool001eoo`03001oogoo0?moo`5oo`007Goo0P00ogoo0Woo000MOol00`00Oomoo`3o
Ool1Ool001eoo`03001oogoo0?moo`5oo`007Goo00<007ooOol0ogoo0Goo000MOol00`00Oomoo`3o
Ool1Ool001eoo`03001oogoo0?moo`5oo`002Goo0P0017oo00<007ooOol00Woo10001Goo00<007oo
Ool0ogoo0Goo0008Ool01000Oomoo`0027oo00<007ooOol01Woo00<007ooOol0ogoo0Goo0008Ool0
1000Oomoo`002Goo00<007ooOol01Goo0P00ogoo0Woo0008Ool01000Oomoo`002Woo00<007ooOol0
ogoo27oo0008Ool01000Oomoo`002goo00<007ooOol0ogoo1goo0008Ool01000Oomoo`0027oo00@0
07ooOol00?moo`Uoo`002Goo0P002Woo0P00ogoo2Woo0000\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {276.312, 99.375}} -> {-0.168621, \
0.108784, 0.00162035, 0.00124846}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
  CellTags->"s:3"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {
            StyleBox["Method",
              "MR"], 
            StyleBox["None",
              
              "MR"], \(Specifies\ the\ method\ to\ use\ with\ fixed\ step\ \(\
\(sizes\)\(.\)\)\)}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  "Options of the method ",
  StyleBox["FixedStep", "MR"],
  ". "
}], "Caption"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.0 for Macintosh",
ScreenRectangle->{{4, 1280}, {0, 1002}},
AutoGeneratedPackage->None,
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{650, 500},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"FixedStep",
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
Magnification->1,
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "c:1"->{
    Cell[1776, 53, 47, 1, 88, "Chapter",
      CellTags->"c:1"]},
  "s:1"->{
    Cell[1848, 58, 50, 1, 61, "Section",
      CellTags->"s:1"]},
  "s:2"->{
    Cell[2903, 98, 46, 1, 61, "Section",
      CellTags->"s:2"]},
  "s:3"->{
    Cell[76027, 2418, 52, 1, 61, "Section",
      CellTags->"s:3"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"c:1", 77469, 2466},
  {"s:1", 77546, 2469},
  {"s:2", 77623, 2472},
  {"s:3", 77700, 2475}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 47, 1, 88, "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{
Cell[1848, 58, 50, 1, 61, "Section",
  CellTags->"s:1"],
Cell[1901, 61, 105, 3, 24, "Text"],
Cell[2009, 66, 337, 8, 56, "Text"],
Cell[2349, 76, 159, 5, 24, "Text"],
Cell[2511, 83, 116, 3, 24, "Text"],
Cell[2630, 88, 236, 5, 42, "Input",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{
Cell[2903, 98, 46, 1, 61, "Section",
  CellTags->"s:2"],
Cell[2952, 101, 42, 0, 24, "Text"],

Cell[CellGroupData[{
Cell[3019, 105, 109, 2, 26, "Input"],
Cell[3131, 109, 834, 19, 90, "Output"]
}, Open  ]],
Cell[3980, 131, 219, 8, 24, "Text"],

Cell[CellGroupData[{
Cell[4224, 143, 289, 6, 58, "Input"],
Cell[4516, 151, 248, 7, 24, "Output"]
}, Open  ]],
Cell[4779, 161, 281, 7, 48, "Text"],

Cell[CellGroupData[{
Cell[5085, 172, 274, 6, 55, "Input"],
Cell[5362, 180, 12000, 480, 184, 5849, 399, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[17377, 663, 140, 4, 21, "Text"],

Cell[CellGroupData[{
Cell[17542, 671, 242, 6, 55, "Input"],
Cell[17787, 679, 25226, 588, 184, 4801, 331, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[43028, 1270, 146, 4, 35, "Text"],

Cell[CellGroupData[{
Cell[43199, 1278, 323, 7, 55, "Input"],
Cell[43525, 1287, 11999, 480, 184, 5849, 399, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[55539, 1770, 174, 5, 35, "Text"],
Cell[55716, 1777, 365, 11, 40, "Text"],

Cell[CellGroupData[{
Cell[56106, 1792, 396, 8, 70, "Input"],
Cell[56505, 1802, 9250, 298, 184, 2882, 214, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[65770, 2103, 219, 5, 40, "Text"],

Cell[CellGroupData[{
Cell[66014, 2112, 380, 8, 74, "Input"],
Cell[66397, 2122, 9581, 290, 184, 2684, 200, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[76027, 2418, 52, 1, 61, "Section",
  CellTags->"s:3"],
Cell[76082, 2421, 464, 12, 62, "DefinitionBox3Col"],
Cell[76549, 2435, 96, 4, 15, "Caption"]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

