(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.0'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[   1259410,      34113]*)
(*NotebookOutlinePosition[   1260942,      34169]*)
(*  CellTagsIndexPosition[   1260761,      34159]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["ImplicitRungeKutta", "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"s:1"],

Cell["\<\
Implicit Runge-Kutta methods have a number of desirable properties.\
\
\>", "Text"],

Cell["\<\
The Gauss-Legendre methods, for example, are self-adjoint meaning \
that they provide the same solution when integrating forwards or backwards in \
time.\
\>", "Text"],

Cell["\<\
This loads packages defining some example problems and utility \
functions.\
\>", "Text"],

Cell[BoxData[{
    \(\(Needs["\<DifferentialEquations`NDSolveProblems`\>"];\)\), "\
\[IndentingNewLine]", 
    \(\(Needs["\<DifferentialEquations`NDSolveUtilities`\>"];\)\)}], "Input",
  CellLabel->"In[3]:=",
  InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Coefficients", "Section",
  CellTags->"s:2"],

Cell["\<\
A generic framework for implicit Runge Kutta methods has been \
implemented. The focus so far is on methods with interesting geometric \
properties and currently covers the following schemes:\
\>", "Text"],

Cell[TextData[{
  StyleBox["ImplicitRungeKuttaGaussCoefficients", "MR"],
  "\n",
  StyleBox["ImplicitRungeKuttaLobattoIIIACoefficients", "MR"],
  "\n",
  StyleBox["ImplicitRungeKuttaLobattoIIIBCoefficients", "MR"],
  "\n",
  StyleBox["ImplicitRungeKuttaLobattoIIICCoefficients", "MR"],
  "\n",
  StyleBox["ImplicitRungeKuttaRadauIACoefficients", "MR"],
  "\n",
  StyleBox["ImplicitRungeKuttaRadauIIACoefficients", "MR"]
}], "Text"],

Cell["\<\
The derivation of the method coefficients can be carried out to \
arbitrary order and arbitrary precision.\
\>", "Text"],

Cell[CellGroupData[{

Cell["Coefficient generation", "Subsection"],

Cell[TextData[{
  "Start with the definition of the polynomial defining the abscissas of the \
",
  Cell[BoxData[
      \(TraditionalForm\`s\)]],
  " stage coefficients. For example, the abscissas for Gauss-Legendre methods \
are defined as ",
  Cell[BoxData[
      \(TraditionalForm\`d\^s\/dx\^s\)]],
  Cell[BoxData[
      FormBox[
        SuperscriptBox[
          RowBox[{
            FormBox[\(x\^s\),
              "TraditionalForm"], "(", \(1 - x\), ")"}], "s"], 
        TraditionalForm]]],
  "."
}], "BulletedList"],

Cell["\<\
Univariate polynomial factorization gives the underlying \
irreducible polynomials defining the roots of the polynomials.\
\>", \
"BulletedList"],

Cell[TextData[{
  StyleBox["Root", "MR"],
  " objects are constructed to represent the solutions (using unique root \
isolation and Jenkins-Traub for the numerical approximation)."
}], "BulletedList"],

Cell[TextData[{
  StyleBox["Root", "MR"],
  " objects are then approximated numerically for precision coefficients."
}], "BulletedList"],

Cell["\<\
Condition estimates for Vandermonde systems governing the \
coefficients yield the precision to take in approximating the roots \
numerically.\
\>", "BulletedList"],

Cell[TextData[{
  "Specialized solvers for nonconfluent Vandermonde systems are then used to \
solve equations for the coefficients (see [",
  ButtonBox["GVL96",
    ButtonData:>{"References.nb", "GVL96"},
    ButtonStyle->"Hyperlink"],
  "])."
}], "BulletedList"],

Cell["\<\
One step of iterative refinement is used to polish the approximate \
solutions and to check that the coefficients are obtained to the requested \
precision.\
\>", "BulletedList"],

Cell["\<\
This generates the coefficients for the two-stage fourth-order \
Gauss-Legendre method to 50-decimal digits of precision.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`ImplicitRungeKuttaGaussCoefficients[4, \ 50]\)], "Input",
  CellLabel->"In[5]:="],

Cell[BoxData[
    \({{{0.25`50. , \
\(-0.038675134594812882254574390250978727823800875635063438009301163242`50. \
\)}, {0.538675134594812882254574390250978727823800875635063438009301163242`50.\
 , 0.25`50. }}, {0.5`50. , 
        0.5`50. }, \
{0.211324865405187117745425609749021272176199124364936561990698836758`50. , 
        0.788675134594812882254574390250978727823800875635063438009301163242`\
50. }}\)], "Output",
  CellLabel->"Out[5]="]
}, Open  ]],

Cell[TextData[{
  "The coefficients have the form ",
  StyleBox["{a,", "MR"],
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`b\^T\)], "MR"], "MR"],
  StyleBox[",", "MR"],
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`c\^T\)], "MR"], "MR"],
  StyleBox["}", "MR"],
  "."
}], "Text"],

Cell["\<\
This generates the coefficients for the two-stage fourth-order \
Gauss-Legendre method exactly. For high-order methods, generating the \
coefficients exactly can often take a very long time.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`ImplicitRungeKuttaGaussCoefficients[4, \ Infinity]\)], "Input",
  CellLabel->"In[6]:="],

Cell[BoxData[
    \({{{1\/4, 1\/12\ \((3 - 2\ \@3)\)}, {1\/12\ \((3 + 2\ \@3)\), 
          1\/4}}, {1\/2, 1\/2}, {1\/6\ \((3 - \@3)\), 
        1\/6\ \((3 + \@3)\)}}\)], "Output",
  CellLabel->"Out[6]="]
}, Open  ]],

Cell["\<\
This generates the coefficients for the six-stage tenth-order \
RaduaIA implicit Runge-Kutta method to 20-decimal digits of precision.\
\>", \
"Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`ImplicitRungeKuttaRadauIACoefficients[10, \ 20]\)], "Input",
  CellLabel->"In[7]:="],

Cell[BoxData[
    \({{{0.04`20. , \(-0.08761801872527423505017362306498456052`20. \), 
          0.08531798763860029375989387300744514058`20. , \
\(-0.05581807848329811483703952661318756651`20. \), 
          0.01811810956997205612731927667072698645`20. }, {0.04`20. , 
          0.12875675325490976115823836749179707516`20. , \
\(-0.04747773040319743429495247634449983651`20. \), 
          0.0267769859677478706880550044955126023`20. , \
\(-0.00829614447567964539925381451792960769`20. \)}, {0.04`20. , 
          0.23310008036710237091548551667442882271`20. , 
          0.16758507013524896344206140916157860597`20. , \
\(-0.03288334354350140177498131770290095956`20. \), 
          0.00860776067223324736073672320397628219`20. }, {0.04`20. , 
          0.21925333267709602304611354942268816425`20. , 
          0.33134489917971587453120843732816154118`20. , 
          0.14621486784749350664968724512404244842`20. , \
\(-0.01365611334242923190705522956051729497`20. \)}, {0.04`20. , 
          0.22493691761630663460402379815045336174`20. , 
          0.30390571559725175839650152645073468243`20. , 
          0.30105430635402060050078494906655890859`20. , 
          0.07299886431790332430556853377813742601`20. }}, {0.04`20. , 
        0.22310390108357074440256021822858935965`20. , 
        0.31182652297574125408185491157664052199`20. , 
        0.2813560151494620601921726503406598912`20. , 
        0.14371356079122594132341221985411022716`20. }, {0, 
        0.13975986434378055215208708112488023326`20. , 
        0.41640956763108317994330233133708275131`20. , 
        0.72315698636187617231995400231437485889`20. , 
        0.94289580388548231780687880744588437876`20. }}\)], "Output",
  CellLabel->"Out[7]="]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
  CellTags->"s:3"],

Cell["Load an example problem.", "Text"],

Cell[BoxData[{
    \(\(system\  = \ 
        GetNDSolveProblem["\<PerturbedKepler\>"];\)\), "\[IndentingNewLine]", \

    \(\(vars\  = \ system["\<DependentVariables\>"[]];\)\)}], "Input",
  CellLabel->"In[8]:="],

Cell["\<\
This problem has two invariants that should remain constant. A \
numerical method may not be able to conserve these invariants.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(invs\  = \ system["\<Invariants\>"[]]\)], "Input",
  CellLabel->"In[10]:="],

Cell[BoxData[
    \({\(-\(1\/\(400\ \((Y\_1[T]\^2 + Y\_2[T]\^2)\)\^\(3/2\)\)\)\) - 
        1\/\@\(Y\_1[T]\^2 + Y\_2[T]\^2\) + 
        1\/2\ \((Y\_3[T]\^2 + Y\_4[T]\^2)\), \(-Y\_2[T]\)\ Y\_3[T] + 
        Y\_1[T]\ Y\_4[T]}\)], "Output",
  CellLabel->"Out[10]="]
}, Open  ]],

Cell[TextData[{
  "This solves the system using an implicit Runge-Kutta Gauss method. The \
order of the scheme is selected using the ",
  StyleBox["DifferenceOrder", "MR"],
  " method option."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol\  = \ 
      NDSolve[system, \ 
        Method \[Rule] {"\<FixedStep\>", \ 
            Method \[Rule] {"\<ImplicitRungeKutta\>", "\<DifferenceOrder\>" \
\[Rule] 10}}, \ StartingStepSize \[Rule] 1/10]\)], "Input",
  CellLabel->"In[11]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_3[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_4[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[11]="]
}, Open  ]],

Cell["\<\
A plot of the error in the invariants shows an increase as the \
integration proceeds.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InvariantErrorPlot[invs, \ vars, \ T, \ sol, \ 
      SymbolStyle \[Rule] {Red, \ Blue}, \ 
      InvariantErrorSampleRate \[Rule] 1]\)], "Input",
  CellLabel->"In[12]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.00952381 0.0238095 4.49303e+09 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.21429 -0.0125 -6 -9 ]
[.21429 -0.0125 6 0 ]
[.40476 -0.0125 -6 -9 ]
[.40476 -0.0125 6 0 ]
[.59524 -0.0125 -6 -9 ]
[.59524 -0.0125 6 0 ]
[.78571 -0.0125 -6 -9 ]
[.78571 -0.0125 6 0 ]
[.97619 -0.0125 -9 -9 ]
[.97619 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .02381 -6 -4.5 ]
[-0.0125 .02381 0 4.5 ]
[-0.0125 .24846 -37.5 -6.25 ]
[-0.0125 .24846 0 6.25 ]
[-0.0125 .47311 -37.5 -6.25 ]
[-0.0125 .47311 0 6.25 ]
[-0.0125 .69776 -49.5 -6.25 ]
[-0.0125 .69776 0 6.25 ]
[-0.0125 .92242 -37.5 -6.25 ]
[-0.0125 .92242 0 6.25 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(20)] .21429 -0.0125 0 1 Mshowa
.40476 0 m
.40476 .00625 L
s
[(40)] .40476 -0.0125 0 1 Mshowa
.59524 0 m
.59524 .00625 L
s
[(60)] .59524 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(80)] .78571 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(100)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07143 0 m
.07143 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.16667 0 m
.16667 .00375 L
s
.2619 0 m
.2619 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.35714 0 m
.35714 .00375 L
s
.45238 0 m
.45238 .00375 L
s
.5 0 m
.5 .00375 L
s
.54762 0 m
.54762 .00375 L
s
.64286 0 m
.64286 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.7381 0 m
.7381 .00375 L
s
.83333 0 m
.83333 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .02381 m
.00625 .02381 L
s
[(0)] -0.0125 .02381 1 0 Mshowa
0 .24846 m
.00625 .24846 L
s
gsave
-0.0125 .24846 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .47311 m
.00625 .47311 L
s
gsave
-0.0125 .47311 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
0 .69776 m
.00625 .69776 L
s
gsave
-0.0125 .69776 -49.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(1.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
0 .92242 m
.00625 .92242 L
s
gsave
-0.0125 .92242 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
.125 Mabswid
0 .06874 m
.00375 .06874 L
s
0 .11367 m
.00375 .11367 L
s
0 .1586 m
.00375 .1586 L
s
0 .20353 m
.00375 .20353 L
s
0 .29339 m
.00375 .29339 L
s
0 .33832 m
.00375 .33832 L
s
0 .38325 m
.00375 .38325 L
s
0 .42818 m
.00375 .42818 L
s
0 .51804 m
.00375 .51804 L
s
0 .56297 m
.00375 .56297 L
s
0 .6079 m
.00375 .6079 L
s
0 .65283 m
.00375 .65283 L
s
0 .74269 m
.00375 .74269 L
s
0 .78762 m
.00375 .78762 L
s
0 .83255 m
.00375 .83255 L
s
0 .87749 m
.00375 .87749 L
s
0 .96735 m
.00375 .96735 L
s
.25 Mabswid
0 0 m
0 1 L
s
.02381 .99375 m
.02381 1 L
s
.21429 .99375 m
.21429 1 L
s
.40476 .99375 m
.40476 1 L
s
.59524 .99375 m
.59524 1 L
s
.78571 .99375 m
.78571 1 L
s
.97619 .99375 m
.97619 1 L
s
.125 Mabswid
.07143 .99625 m
.07143 1 L
s
.11905 .99625 m
.11905 1 L
s
.16667 .99625 m
.16667 1 L
s
.2619 .99625 m
.2619 1 L
s
.30952 .99625 m
.30952 1 L
s
.35714 .99625 m
.35714 1 L
s
.45238 .99625 m
.45238 1 L
s
.5 .99625 m
.5 1 L
s
.54762 .99625 m
.54762 1 L
s
.64286 .99625 m
.64286 1 L
s
.69048 .99625 m
.69048 1 L
s
.7381 .99625 m
.7381 1 L
s
.83333 .99625 m
.83333 1 L
s
.88095 .99625 m
.88095 1 L
s
.92857 .99625 m
.92857 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .02381 m
1 .02381 L
s
.99375 .24846 m
1 .24846 L
s
.99375 .47311 m
1 .47311 L
s
.99375 .69776 m
1 .69776 L
s
.99375 .92242 m
1 .92242 L
s
.125 Mabswid
.99625 .06874 m
1 .06874 L
s
.99625 .11367 m
1 .11367 L
s
.99625 .1586 m
1 .1586 L
s
.99625 .20353 m
1 .20353 L
s
.99625 .29339 m
1 .29339 L
s
.99625 .33832 m
1 .33832 L
s
.99625 .38325 m
1 .38325 L
s
.99625 .42818 m
1 .42818 L
s
.99625 .51804 m
1 .51804 L
s
.99625 .56297 m
1 .56297 L
s
.99625 .6079 m
1 .6079 L
s
.99625 .65283 m
1 .65283 L
s
.99625 .74269 m
1 .74269 L
s
.99625 .78762 m
1 .78762 L
s
.99625 .83255 m
1 .83255 L
s
.99625 .87749 m
1 .87749 L
s
.99625 .96735 m
1 .96735 L
s
.25 Mabswid
1 0 m
1 1 L
s
1 0 0 r
.0001 w
.02381 .02381 -1.875 0 Mabsadd m
.02381 .02381 0 2.5 Mabsadd L
.02381 .02381 1.875 0 Mabsadd L
.02381 .02381 0 -2.5 Mabsadd L
.02381 .02381 -1.875 0 Mabsadd L closepath F
.02476 .20582 -1.875 0 Mabsadd m
.02476 .20582 0 2.5 Mabsadd L
.02476 .20582 1.875 0 Mabsadd L
.02476 .20582 0 -2.5 Mabsadd L
.02476 .20582 -1.875 0 Mabsadd L closepath F
.02571 .19747 -1.875 0 Mabsadd m
.02571 .19747 0 2.5 Mabsadd L
.02571 .19747 1.875 0 Mabsadd L
.02571 .19747 0 -2.5 Mabsadd L
.02571 .19747 -1.875 0 Mabsadd L closepath F
.02667 .2177 -1.875 0 Mabsadd m
.02667 .2177 0 2.5 Mabsadd L
.02667 .2177 1.875 0 Mabsadd L
.02667 .2177 0 -2.5 Mabsadd L
.02667 .2177 -1.875 0 Mabsadd L closepath F
.02762 .21738 -1.875 0 Mabsadd m
.02762 .21738 0 2.5 Mabsadd L
.02762 .21738 1.875 0 Mabsadd L
.02762 .21738 0 -2.5 Mabsadd L
.02762 .21738 -1.875 0 Mabsadd L closepath F
.02857 .21837 -1.875 0 Mabsadd m
.02857 .21837 0 2.5 Mabsadd L
.02857 .21837 1.875 0 Mabsadd L
.02857 .21837 0 -2.5 Mabsadd L
.02857 .21837 -1.875 0 Mabsadd L closepath F
.02952 .22381 -1.875 0 Mabsadd m
.02952 .22381 0 2.5 Mabsadd L
.02952 .22381 1.875 0 Mabsadd L
.02952 .22381 0 -2.5 Mabsadd L
.02952 .22381 -1.875 0 Mabsadd L closepath F
.03048 .22863 -1.875 0 Mabsadd m
.03048 .22863 0 2.5 Mabsadd L
.03048 .22863 1.875 0 Mabsadd L
.03048 .22863 0 -2.5 Mabsadd L
.03048 .22863 -1.875 0 Mabsadd L closepath F
.03143 .26247 -1.875 0 Mabsadd m
.03143 .26247 0 2.5 Mabsadd L
.03143 .26247 1.875 0 Mabsadd L
.03143 .26247 0 -2.5 Mabsadd L
.03143 .26247 -1.875 0 Mabsadd L closepath F
.03238 .27024 -1.875 0 Mabsadd m
.03238 .27024 0 2.5 Mabsadd L
.03238 .27024 1.875 0 Mabsadd L
.03238 .27024 0 -2.5 Mabsadd L
.03238 .27024 -1.875 0 Mabsadd L closepath F
.03333 .27216 -1.875 0 Mabsadd m
.03333 .27216 0 2.5 Mabsadd L
.03333 .27216 1.875 0 Mabsadd L
.03333 .27216 0 -2.5 Mabsadd L
.03333 .27216 -1.875 0 Mabsadd L closepath F
.03429 .27644 -1.875 0 Mabsadd m
.03429 .27644 0 2.5 Mabsadd L
.03429 .27644 1.875 0 Mabsadd L
.03429 .27644 0 -2.5 Mabsadd L
.03429 .27644 -1.875 0 Mabsadd L closepath F
.03524 .27662 -1.875 0 Mabsadd m
.03524 .27662 0 2.5 Mabsadd L
.03524 .27662 1.875 0 Mabsadd L
.03524 .27662 0 -2.5 Mabsadd L
.03524 .27662 -1.875 0 Mabsadd L closepath F
.03619 .27709 -1.875 0 Mabsadd m
.03619 .27709 0 2.5 Mabsadd L
.03619 .27709 1.875 0 Mabsadd L
.03619 .27709 0 -2.5 Mabsadd L
.03619 .27709 -1.875 0 Mabsadd L closepath F
.03714 .27712 -1.875 0 Mabsadd m
.03714 .27712 0 2.5 Mabsadd L
.03714 .27712 1.875 0 Mabsadd L
.03714 .27712 0 -2.5 Mabsadd L
.03714 .27712 -1.875 0 Mabsadd L closepath F
.0381 .2772 -1.875 0 Mabsadd m
.0381 .2772 0 2.5 Mabsadd L
.0381 .2772 1.875 0 Mabsadd L
.0381 .2772 0 -2.5 Mabsadd L
.0381 .2772 -1.875 0 Mabsadd L closepath F
.03905 .2772 -1.875 0 Mabsadd m
.03905 .2772 0 2.5 Mabsadd L
.03905 .2772 1.875 0 Mabsadd L
.03905 .2772 0 -2.5 Mabsadd L
.03905 .2772 -1.875 0 Mabsadd L closepath F
.04 .27722 -1.875 0 Mabsadd m
.04 .27722 0 2.5 Mabsadd L
.04 .27722 1.875 0 Mabsadd L
.04 .27722 0 -2.5 Mabsadd L
.04 .27722 -1.875 0 Mabsadd L closepath F
.04095 .27722 -1.875 0 Mabsadd m
.04095 .27722 0 2.5 Mabsadd L
.04095 .27722 1.875 0 Mabsadd L
.04095 .27722 0 -2.5 Mabsadd L
.04095 .27722 -1.875 0 Mabsadd L closepath F
.0419 .27723 -1.875 0 Mabsadd m
.0419 .27723 0 2.5 Mabsadd L
.0419 .27723 1.875 0 Mabsadd L
.0419 .27723 0 -2.5 Mabsadd L
.0419 .27723 -1.875 0 Mabsadd L closepath F
.04286 .27724 -1.875 0 Mabsadd m
.04286 .27724 0 2.5 Mabsadd L
.04286 .27724 1.875 0 Mabsadd L
.04286 .27724 0 -2.5 Mabsadd L
.04286 .27724 -1.875 0 Mabsadd L closepath F
.04381 .27724 -1.875 0 Mabsadd m
.04381 .27724 0 2.5 Mabsadd L
.04381 .27724 1.875 0 Mabsadd L
.04381 .27724 0 -2.5 Mabsadd L
.04381 .27724 -1.875 0 Mabsadd L closepath F
.04476 .27724 -1.875 0 Mabsadd m
.04476 .27724 0 2.5 Mabsadd L
.04476 .27724 1.875 0 Mabsadd L
.04476 .27724 0 -2.5 Mabsadd L
.04476 .27724 -1.875 0 Mabsadd L closepath F
.04571 .27724 -1.875 0 Mabsadd m
.04571 .27724 0 2.5 Mabsadd L
.04571 .27724 1.875 0 Mabsadd L
.04571 .27724 0 -2.5 Mabsadd L
.04571 .27724 -1.875 0 Mabsadd L closepath F
.04667 .27725 -1.875 0 Mabsadd m
.04667 .27725 0 2.5 Mabsadd L
.04667 .27725 1.875 0 Mabsadd L
.04667 .27725 0 -2.5 Mabsadd L
.04667 .27725 -1.875 0 Mabsadd L closepath F
.04762 .27725 -1.875 0 Mabsadd m
.04762 .27725 0 2.5 Mabsadd L
.04762 .27725 1.875 0 Mabsadd L
.04762 .27725 0 -2.5 Mabsadd L
.04762 .27725 -1.875 0 Mabsadd L closepath F
.04857 .27725 -1.875 0 Mabsadd m
.04857 .27725 0 2.5 Mabsadd L
.04857 .27725 1.875 0 Mabsadd L
.04857 .27725 0 -2.5 Mabsadd L
.04857 .27725 -1.875 0 Mabsadd L closepath F
.04952 .27725 -1.875 0 Mabsadd m
.04952 .27725 0 2.5 Mabsadd L
.04952 .27725 1.875 0 Mabsadd L
.04952 .27725 0 -2.5 Mabsadd L
.04952 .27725 -1.875 0 Mabsadd L closepath F
.05048 .27725 -1.875 0 Mabsadd m
.05048 .27725 0 2.5 Mabsadd L
.05048 .27725 1.875 0 Mabsadd L
.05048 .27725 0 -2.5 Mabsadd L
.05048 .27725 -1.875 0 Mabsadd L closepath F
.05143 .27725 -1.875 0 Mabsadd m
.05143 .27725 0 2.5 Mabsadd L
.05143 .27725 1.875 0 Mabsadd L
.05143 .27725 0 -2.5 Mabsadd L
.05143 .27725 -1.875 0 Mabsadd L closepath F
.05238 .27725 -1.875 0 Mabsadd m
.05238 .27725 0 2.5 Mabsadd L
.05238 .27725 1.875 0 Mabsadd L
.05238 .27725 0 -2.5 Mabsadd L
.05238 .27725 -1.875 0 Mabsadd L closepath F
.05333 .27725 -1.875 0 Mabsadd m
.05333 .27725 0 2.5 Mabsadd L
.05333 .27725 1.875 0 Mabsadd L
.05333 .27725 0 -2.5 Mabsadd L
.05333 .27725 -1.875 0 Mabsadd L closepath F
.05429 .27724 -1.875 0 Mabsadd m
.05429 .27724 0 2.5 Mabsadd L
.05429 .27724 1.875 0 Mabsadd L
.05429 .27724 0 -2.5 Mabsadd L
.05429 .27724 -1.875 0 Mabsadd L closepath F
.05524 .27724 -1.875 0 Mabsadd m
.05524 .27724 0 2.5 Mabsadd L
.05524 .27724 1.875 0 Mabsadd L
.05524 .27724 0 -2.5 Mabsadd L
.05524 .27724 -1.875 0 Mabsadd L closepath F
.05619 .27724 -1.875 0 Mabsadd m
.05619 .27724 0 2.5 Mabsadd L
.05619 .27724 1.875 0 Mabsadd L
.05619 .27724 0 -2.5 Mabsadd L
.05619 .27724 -1.875 0 Mabsadd L closepath F
.05714 .27724 -1.875 0 Mabsadd m
.05714 .27724 0 2.5 Mabsadd L
.05714 .27724 1.875 0 Mabsadd L
.05714 .27724 0 -2.5 Mabsadd L
.05714 .27724 -1.875 0 Mabsadd L closepath F
.0581 .27724 -1.875 0 Mabsadd m
.0581 .27724 0 2.5 Mabsadd L
.0581 .27724 1.875 0 Mabsadd L
.0581 .27724 0 -2.5 Mabsadd L
.0581 .27724 -1.875 0 Mabsadd L closepath F
.05905 .27724 -1.875 0 Mabsadd m
.05905 .27724 0 2.5 Mabsadd L
.05905 .27724 1.875 0 Mabsadd L
.05905 .27724 0 -2.5 Mabsadd L
.05905 .27724 -1.875 0 Mabsadd L closepath F
.06 .27724 -1.875 0 Mabsadd m
.06 .27724 0 2.5 Mabsadd L
.06 .27724 1.875 0 Mabsadd L
.06 .27724 0 -2.5 Mabsadd L
.06 .27724 -1.875 0 Mabsadd L closepath F
.06095 .27723 -1.875 0 Mabsadd m
.06095 .27723 0 2.5 Mabsadd L
.06095 .27723 1.875 0 Mabsadd L
.06095 .27723 0 -2.5 Mabsadd L
.06095 .27723 -1.875 0 Mabsadd L closepath F
.0619 .27723 -1.875 0 Mabsadd m
.0619 .27723 0 2.5 Mabsadd L
.0619 .27723 1.875 0 Mabsadd L
.0619 .27723 0 -2.5 Mabsadd L
.0619 .27723 -1.875 0 Mabsadd L closepath F
.06286 .27722 -1.875 0 Mabsadd m
.06286 .27722 0 2.5 Mabsadd L
.06286 .27722 1.875 0 Mabsadd L
.06286 .27722 0 -2.5 Mabsadd L
.06286 .27722 -1.875 0 Mabsadd L closepath F
.06381 .27721 -1.875 0 Mabsadd m
.06381 .27721 0 2.5 Mabsadd L
.06381 .27721 1.875 0 Mabsadd L
.06381 .27721 0 -2.5 Mabsadd L
.06381 .27721 -1.875 0 Mabsadd L closepath F
.06476 .27718 -1.875 0 Mabsadd m
.06476 .27718 0 2.5 Mabsadd L
.06476 .27718 1.875 0 Mabsadd L
.06476 .27718 0 -2.5 Mabsadd L
.06476 .27718 -1.875 0 Mabsadd L closepath F
.06571 .27716 -1.875 0 Mabsadd m
.06571 .27716 0 2.5 Mabsadd L
.06571 .27716 1.875 0 Mabsadd L
.06571 .27716 0 -2.5 Mabsadd L
.06571 .27716 -1.875 0 Mabsadd L closepath F
.06667 .27697 -1.875 0 Mabsadd m
.06667 .27697 0 2.5 Mabsadd L
.06667 .27697 1.875 0 Mabsadd L
.06667 .27697 0 -2.5 Mabsadd L
.06667 .27697 -1.875 0 Mabsadd L closepath F
.06762 .27684 -1.875 0 Mabsadd m
.06762 .27684 0 2.5 Mabsadd L
.06762 .27684 1.875 0 Mabsadd L
.06762 .27684 0 -2.5 Mabsadd L
.06762 .27684 -1.875 0 Mabsadd L closepath F
.06857 .27524 -1.875 0 Mabsadd m
.06857 .27524 0 2.5 Mabsadd L
.06857 .27524 1.875 0 Mabsadd L
.06857 .27524 0 -2.5 Mabsadd L
.06857 .27524 -1.875 0 Mabsadd L closepath F
.06952 .27386 -1.875 0 Mabsadd m
.06952 .27386 0 2.5 Mabsadd L
.06952 .27386 1.875 0 Mabsadd L
.06952 .27386 0 -2.5 Mabsadd L
.06952 .27386 -1.875 0 Mabsadd L closepath F
.07048 .26798 -1.875 0 Mabsadd m
.07048 .26798 0 2.5 Mabsadd L
.07048 .26798 1.875 0 Mabsadd L
.07048 .26798 0 -2.5 Mabsadd L
.07048 .26798 -1.875 0 Mabsadd L closepath F
.07143 .2389 -1.875 0 Mabsadd m
.07143 .2389 0 2.5 Mabsadd L
.07143 .2389 1.875 0 Mabsadd L
.07143 .2389 0 -2.5 Mabsadd L
.07143 .2389 -1.875 0 Mabsadd L closepath F
.07238 .23889 -1.875 0 Mabsadd m
.07238 .23889 0 2.5 Mabsadd L
.07238 .23889 1.875 0 Mabsadd L
.07238 .23889 0 -2.5 Mabsadd L
.07238 .23889 -1.875 0 Mabsadd L closepath F
.07333 .24014 -1.875 0 Mabsadd m
.07333 .24014 0 2.5 Mabsadd L
.07333 .24014 1.875 0 Mabsadd L
.07333 .24014 0 -2.5 Mabsadd L
.07333 .24014 -1.875 0 Mabsadd L closepath F
.07429 .25193 -1.875 0 Mabsadd m
.07429 .25193 0 2.5 Mabsadd L
.07429 .25193 1.875 0 Mabsadd L
.07429 .25193 0 -2.5 Mabsadd L
.07429 .25193 -1.875 0 Mabsadd L closepath F
.07524 .27231 -1.875 0 Mabsadd m
.07524 .27231 0 2.5 Mabsadd L
.07524 .27231 1.875 0 Mabsadd L
.07524 .27231 0 -2.5 Mabsadd L
.07524 .27231 -1.875 0 Mabsadd L closepath F
.07619 .26783 -1.875 0 Mabsadd m
.07619 .26783 0 2.5 Mabsadd L
.07619 .26783 1.875 0 Mabsadd L
.07619 .26783 0 -2.5 Mabsadd L
.07619 .26783 -1.875 0 Mabsadd L closepath F
.07714 .12114 -1.875 0 Mabsadd m
.07714 .12114 0 2.5 Mabsadd L
.07714 .12114 1.875 0 Mabsadd L
.07714 .12114 0 -2.5 Mabsadd L
.07714 .12114 -1.875 0 Mabsadd L closepath F
.0781 .32838 -1.875 0 Mabsadd m
.0781 .32838 0 2.5 Mabsadd L
.0781 .32838 1.875 0 Mabsadd L
.0781 .32838 0 -2.5 Mabsadd L
.0781 .32838 -1.875 0 Mabsadd L closepath F
.07905 .31058 -1.875 0 Mabsadd m
.07905 .31058 0 2.5 Mabsadd L
.07905 .31058 1.875 0 Mabsadd L
.07905 .31058 0 -2.5 Mabsadd L
.07905 .31058 -1.875 0 Mabsadd L closepath F
.08 .33479 -1.875 0 Mabsadd m
.08 .33479 0 2.5 Mabsadd L
.08 .33479 1.875 0 Mabsadd L
.08 .33479 0 -2.5 Mabsadd L
.08 .33479 -1.875 0 Mabsadd L closepath F
.08095 .33408 -1.875 0 Mabsadd m
.08095 .33408 0 2.5 Mabsadd L
.08095 .33408 1.875 0 Mabsadd L
.08095 .33408 0 -2.5 Mabsadd L
.08095 .33408 -1.875 0 Mabsadd L closepath F
.0819 .33541 -1.875 0 Mabsadd m
.0819 .33541 0 2.5 Mabsadd L
.0819 .33541 1.875 0 Mabsadd L
.0819 .33541 0 -2.5 Mabsadd L
.0819 .33541 -1.875 0 Mabsadd L closepath F
.08286 .34296 -1.875 0 Mabsadd m
.08286 .34296 0 2.5 Mabsadd L
.08286 .34296 1.875 0 Mabsadd L
.08286 .34296 0 -2.5 Mabsadd L
.08286 .34296 -1.875 0 Mabsadd L closepath F
.08381 .343 -1.875 0 Mabsadd m
.08381 .343 0 2.5 Mabsadd L
.08381 .343 1.875 0 Mabsadd L
.08381 .343 0 -2.5 Mabsadd L
.08381 .343 -1.875 0 Mabsadd L closepath F
.08476 .34309 -1.875 0 Mabsadd m
.08476 .34309 0 2.5 Mabsadd L
.08476 .34309 1.875 0 Mabsadd L
.08476 .34309 0 -2.5 Mabsadd L
.08476 .34309 -1.875 0 Mabsadd L closepath F
.08571 .34319 -1.875 0 Mabsadd m
.08571 .34319 0 2.5 Mabsadd L
.08571 .34319 1.875 0 Mabsadd L
.08571 .34319 0 -2.5 Mabsadd L
.08571 .34319 -1.875 0 Mabsadd L closepath F
.08667 .34325 -1.875 0 Mabsadd m
.08667 .34325 0 2.5 Mabsadd L
.08667 .34325 1.875 0 Mabsadd L
.08667 .34325 0 -2.5 Mabsadd L
.08667 .34325 -1.875 0 Mabsadd L closepath F
.08762 .34328 -1.875 0 Mabsadd m
.08762 .34328 0 2.5 Mabsadd L
.08762 .34328 1.875 0 Mabsadd L
.08762 .34328 0 -2.5 Mabsadd L
.08762 .34328 -1.875 0 Mabsadd L closepath F
.08857 .34348 -1.875 0 Mabsadd m
.08857 .34348 0 2.5 Mabsadd L
.08857 .34348 1.875 0 Mabsadd L
.08857 .34348 0 -2.5 Mabsadd L
.08857 .34348 -1.875 0 Mabsadd L closepath F
.08952 .34402 -1.875 0 Mabsadd m
.08952 .34402 0 2.5 Mabsadd L
.08952 .34402 1.875 0 Mabsadd L
.08952 .34402 0 -2.5 Mabsadd L
.08952 .34402 -1.875 0 Mabsadd L closepath F
.09048 .34405 -1.875 0 Mabsadd m
.09048 .34405 0 2.5 Mabsadd L
.09048 .34405 1.875 0 Mabsadd L
.09048 .34405 0 -2.5 Mabsadd L
.09048 .34405 -1.875 0 Mabsadd L closepath F
.09143 .34414 -1.875 0 Mabsadd m
.09143 .34414 0 2.5 Mabsadd L
.09143 .34414 1.875 0 Mabsadd L
.09143 .34414 0 -2.5 Mabsadd L
.09143 .34414 -1.875 0 Mabsadd L closepath F
.09238 .34414 -1.875 0 Mabsadd m
.09238 .34414 0 2.5 Mabsadd L
.09238 .34414 1.875 0 Mabsadd L
.09238 .34414 0 -2.5 Mabsadd L
.09238 .34414 -1.875 0 Mabsadd L closepath F
.09333 .34416 -1.875 0 Mabsadd m
.09333 .34416 0 2.5 Mabsadd L
.09333 .34416 1.875 0 Mabsadd L
.09333 .34416 0 -2.5 Mabsadd L
.09333 .34416 -1.875 0 Mabsadd L closepath F
.09429 .34416 -1.875 0 Mabsadd m
.09429 .34416 0 2.5 Mabsadd L
.09429 .34416 1.875 0 Mabsadd L
.09429 .34416 0 -2.5 Mabsadd L
.09429 .34416 -1.875 0 Mabsadd L closepath F
.09524 .34417 -1.875 0 Mabsadd m
.09524 .34417 0 2.5 Mabsadd L
.09524 .34417 1.875 0 Mabsadd L
.09524 .34417 0 -2.5 Mabsadd L
.09524 .34417 -1.875 0 Mabsadd L closepath F
.09619 .34418 -1.875 0 Mabsadd m
.09619 .34418 0 2.5 Mabsadd L
.09619 .34418 1.875 0 Mabsadd L
.09619 .34418 0 -2.5 Mabsadd L
.09619 .34418 -1.875 0 Mabsadd L closepath F
.09714 .34418 -1.875 0 Mabsadd m
.09714 .34418 0 2.5 Mabsadd L
.09714 .34418 1.875 0 Mabsadd L
.09714 .34418 0 -2.5 Mabsadd L
.09714 .34418 -1.875 0 Mabsadd L closepath F
.0981 .34418 -1.875 0 Mabsadd m
.0981 .34418 0 2.5 Mabsadd L
.0981 .34418 1.875 0 Mabsadd L
.0981 .34418 0 -2.5 Mabsadd L
.0981 .34418 -1.875 0 Mabsadd L closepath F
.09905 .34418 -1.875 0 Mabsadd m
.09905 .34418 0 2.5 Mabsadd L
.09905 .34418 1.875 0 Mabsadd L
.09905 .34418 0 -2.5 Mabsadd L
.09905 .34418 -1.875 0 Mabsadd L closepath F
.1 .34419 -1.875 0 Mabsadd m
.1 .34419 0 2.5 Mabsadd L
.1 .34419 1.875 0 Mabsadd L
.1 .34419 0 -2.5 Mabsadd L
.1 .34419 -1.875 0 Mabsadd L closepath F
.10095 .34419 -1.875 0 Mabsadd m
.10095 .34419 0 2.5 Mabsadd L
.10095 .34419 1.875 0 Mabsadd L
.10095 .34419 0 -2.5 Mabsadd L
.10095 .34419 -1.875 0 Mabsadd L closepath F
.1019 .34419 -1.875 0 Mabsadd m
.1019 .34419 0 2.5 Mabsadd L
.1019 .34419 1.875 0 Mabsadd L
.1019 .34419 0 -2.5 Mabsadd L
.1019 .34419 -1.875 0 Mabsadd L closepath F
.10286 .34419 -1.875 0 Mabsadd m
.10286 .34419 0 2.5 Mabsadd L
.10286 .34419 1.875 0 Mabsadd L
.10286 .34419 0 -2.5 Mabsadd L
.10286 .34419 -1.875 0 Mabsadd L closepath F
.10381 .34419 -1.875 0 Mabsadd m
.10381 .34419 0 2.5 Mabsadd L
.10381 .34419 1.875 0 Mabsadd L
.10381 .34419 0 -2.5 Mabsadd L
.10381 .34419 -1.875 0 Mabsadd L closepath F
.10476 .34419 -1.875 0 Mabsadd m
.10476 .34419 0 2.5 Mabsadd L
.10476 .34419 1.875 0 Mabsadd L
.10476 .34419 0 -2.5 Mabsadd L
.10476 .34419 -1.875 0 Mabsadd L closepath F
.10571 .34419 -1.875 0 Mabsadd m
.10571 .34419 0 2.5 Mabsadd L
.10571 .34419 1.875 0 Mabsadd L
.10571 .34419 0 -2.5 Mabsadd L
.10571 .34419 -1.875 0 Mabsadd L closepath F
.10667 .34419 -1.875 0 Mabsadd m
.10667 .34419 0 2.5 Mabsadd L
.10667 .34419 1.875 0 Mabsadd L
.10667 .34419 0 -2.5 Mabsadd L
.10667 .34419 -1.875 0 Mabsadd L closepath F
.10762 .34419 -1.875 0 Mabsadd m
.10762 .34419 0 2.5 Mabsadd L
.10762 .34419 1.875 0 Mabsadd L
.10762 .34419 0 -2.5 Mabsadd L
.10762 .34419 -1.875 0 Mabsadd L closepath F
.10857 .34419 -1.875 0 Mabsadd m
.10857 .34419 0 2.5 Mabsadd L
.10857 .34419 1.875 0 Mabsadd L
.10857 .34419 0 -2.5 Mabsadd L
.10857 .34419 -1.875 0 Mabsadd L closepath F
.10952 .34418 -1.875 0 Mabsadd m
.10952 .34418 0 2.5 Mabsadd L
.10952 .34418 1.875 0 Mabsadd L
.10952 .34418 0 -2.5 Mabsadd L
.10952 .34418 -1.875 0 Mabsadd L closepath F
.11048 .34418 -1.875 0 Mabsadd m
.11048 .34418 0 2.5 Mabsadd L
.11048 .34418 1.875 0 Mabsadd L
.11048 .34418 0 -2.5 Mabsadd L
.11048 .34418 -1.875 0 Mabsadd L closepath F
.11143 .34418 -1.875 0 Mabsadd m
.11143 .34418 0 2.5 Mabsadd L
.11143 .34418 1.875 0 Mabsadd L
.11143 .34418 0 -2.5 Mabsadd L
.11143 .34418 -1.875 0 Mabsadd L closepath F
.11238 .34418 -1.875 0 Mabsadd m
.11238 .34418 0 2.5 Mabsadd L
.11238 .34418 1.875 0 Mabsadd L
.11238 .34418 0 -2.5 Mabsadd L
.11238 .34418 -1.875 0 Mabsadd L closepath F
.11333 .34418 -1.875 0 Mabsadd m
.11333 .34418 0 2.5 Mabsadd L
.11333 .34418 1.875 0 Mabsadd L
.11333 .34418 0 -2.5 Mabsadd L
.11333 .34418 -1.875 0 Mabsadd L closepath F
.11429 .34417 -1.875 0 Mabsadd m
.11429 .34417 0 2.5 Mabsadd L
.11429 .34417 1.875 0 Mabsadd L
.11429 .34417 0 -2.5 Mabsadd L
.11429 .34417 -1.875 0 Mabsadd L closepath F
.11524 .34417 -1.875 0 Mabsadd m
.11524 .34417 0 2.5 Mabsadd L
.11524 .34417 1.875 0 Mabsadd L
.11524 .34417 0 -2.5 Mabsadd L
.11524 .34417 -1.875 0 Mabsadd L closepath F
.11619 .34416 -1.875 0 Mabsadd m
.11619 .34416 0 2.5 Mabsadd L
.11619 .34416 1.875 0 Mabsadd L
.11619 .34416 0 -2.5 Mabsadd L
.11619 .34416 -1.875 0 Mabsadd L closepath F
.11714 .34416 -1.875 0 Mabsadd m
.11714 .34416 0 2.5 Mabsadd L
.11714 .34416 1.875 0 Mabsadd L
.11714 .34416 0 -2.5 Mabsadd L
.11714 .34416 -1.875 0 Mabsadd L closepath F
.1181 .34412 -1.875 0 Mabsadd m
.1181 .34412 0 2.5 Mabsadd L
.1181 .34412 1.875 0 Mabsadd L
.1181 .34412 0 -2.5 Mabsadd L
.1181 .34412 -1.875 0 Mabsadd L closepath F
.11905 .34411 -1.875 0 Mabsadd m
.11905 .34411 0 2.5 Mabsadd L
.11905 .34411 1.875 0 Mabsadd L
.11905 .34411 0 -2.5 Mabsadd L
.11905 .34411 -1.875 0 Mabsadd L closepath F
.12 .34394 -1.875 0 Mabsadd m
.12 .34394 0 2.5 Mabsadd L
.12 .34394 1.875 0 Mabsadd L
.12 .34394 0 -2.5 Mabsadd L
.12 .34394 -1.875 0 Mabsadd L closepath F
.12095 .34383 -1.875 0 Mabsadd m
.12095 .34383 0 2.5 Mabsadd L
.12095 .34383 1.875 0 Mabsadd L
.12095 .34383 0 -2.5 Mabsadd L
.12095 .34383 -1.875 0 Mabsadd L closepath F
.1219 .34246 -1.875 0 Mabsadd m
.1219 .34246 0 2.5 Mabsadd L
.1219 .34246 1.875 0 Mabsadd L
.1219 .34246 0 -2.5 Mabsadd L
.1219 .34246 -1.875 0 Mabsadd L closepath F
.12286 .3413 -1.875 0 Mabsadd m
.12286 .3413 0 2.5 Mabsadd L
.12286 .3413 1.875 0 Mabsadd L
.12286 .3413 0 -2.5 Mabsadd L
.12286 .3413 -1.875 0 Mabsadd L closepath F
.12381 .32211 -1.875 0 Mabsadd m
.12381 .32211 0 2.5 Mabsadd L
.12381 .32211 1.875 0 Mabsadd L
.12381 .32211 0 -2.5 Mabsadd L
.12381 .32211 -1.875 0 Mabsadd L closepath F
.12476 .29898 -1.875 0 Mabsadd m
.12476 .29898 0 2.5 Mabsadd L
.12476 .29898 1.875 0 Mabsadd L
.12476 .29898 0 -2.5 Mabsadd L
.12476 .29898 -1.875 0 Mabsadd L closepath F
.12571 .29896 -1.875 0 Mabsadd m
.12571 .29896 0 2.5 Mabsadd L
.12571 .29896 1.875 0 Mabsadd L
.12571 .29896 0 -2.5 Mabsadd L
.12571 .29896 -1.875 0 Mabsadd L closepath F
.12667 .29992 -1.875 0 Mabsadd m
.12667 .29992 0 2.5 Mabsadd L
.12667 .29992 1.875 0 Mabsadd L
.12667 .29992 0 -2.5 Mabsadd L
.12667 .29992 -1.875 0 Mabsadd L closepath F
.12762 .30967 -1.875 0 Mabsadd m
.12762 .30967 0 2.5 Mabsadd L
.12762 .30967 1.875 0 Mabsadd L
.12762 .30967 0 -2.5 Mabsadd L
.12762 .30967 -1.875 0 Mabsadd L closepath F
.12857 .32267 -1.875 0 Mabsadd m
.12857 .32267 0 2.5 Mabsadd L
.12857 .32267 1.875 0 Mabsadd L
.12857 .32267 0 -2.5 Mabsadd L
.12857 .32267 -1.875 0 Mabsadd L closepath F
.12952 .30586 -1.875 0 Mabsadd m
.12952 .30586 0 2.5 Mabsadd L
.12952 .30586 1.875 0 Mabsadd L
.12952 .30586 0 -2.5 Mabsadd L
.12952 .30586 -1.875 0 Mabsadd L closepath F
.13048 .20808 -1.875 0 Mabsadd m
.13048 .20808 0 2.5 Mabsadd L
.13048 .20808 1.875 0 Mabsadd L
.13048 .20808 0 -2.5 Mabsadd L
.13048 .20808 -1.875 0 Mabsadd L closepath F
.13143 .37857 -1.875 0 Mabsadd m
.13143 .37857 0 2.5 Mabsadd L
.13143 .37857 1.875 0 Mabsadd L
.13143 .37857 0 -2.5 Mabsadd L
.13143 .37857 -1.875 0 Mabsadd L closepath F
.13238 .35996 -1.875 0 Mabsadd m
.13238 .35996 0 2.5 Mabsadd L
.13238 .35996 1.875 0 Mabsadd L
.13238 .35996 0 -2.5 Mabsadd L
.13238 .35996 -1.875 0 Mabsadd L closepath F
.13333 .3873 -1.875 0 Mabsadd m
.13333 .3873 0 2.5 Mabsadd L
.13333 .3873 1.875 0 Mabsadd L
.13333 .3873 0 -2.5 Mabsadd L
.13333 .3873 -1.875 0 Mabsadd L closepath F
.13429 .38585 -1.875 0 Mabsadd m
.13429 .38585 0 2.5 Mabsadd L
.13429 .38585 1.875 0 Mabsadd L
.13429 .38585 0 -2.5 Mabsadd L
.13429 .38585 -1.875 0 Mabsadd L closepath F
.13524 .38765 -1.875 0 Mabsadd m
.13524 .38765 0 2.5 Mabsadd L
.13524 .38765 1.875 0 Mabsadd L
.13524 .38765 0 -2.5 Mabsadd L
.13524 .38765 -1.875 0 Mabsadd L closepath F
.13619 .3982 -1.875 0 Mabsadd m
.13619 .3982 0 2.5 Mabsadd L
.13619 .3982 1.875 0 Mabsadd L
.13619 .3982 0 -2.5 Mabsadd L
.13619 .3982 -1.875 0 Mabsadd L closepath F
.13714 .39825 -1.875 0 Mabsadd m
.13714 .39825 0 2.5 Mabsadd L
.13714 .39825 1.875 0 Mabsadd L
.13714 .39825 0 -2.5 Mabsadd L
.13714 .39825 -1.875 0 Mabsadd L closepath F
.1381 .39836 -1.875 0 Mabsadd m
.1381 .39836 0 2.5 Mabsadd L
.1381 .39836 1.875 0 Mabsadd L
.1381 .39836 0 -2.5 Mabsadd L
.1381 .39836 -1.875 0 Mabsadd L closepath F
.13905 .39849 -1.875 0 Mabsadd m
.13905 .39849 0 2.5 Mabsadd L
.13905 .39849 1.875 0 Mabsadd L
.13905 .39849 0 -2.5 Mabsadd L
.13905 .39849 -1.875 0 Mabsadd L closepath F
.14 .39857 -1.875 0 Mabsadd m
.14 .39857 0 2.5 Mabsadd L
.14 .39857 1.875 0 Mabsadd L
.14 .39857 0 -2.5 Mabsadd L
.14 .39857 -1.875 0 Mabsadd L closepath F
.14095 .39861 -1.875 0 Mabsadd m
.14095 .39861 0 2.5 Mabsadd L
.14095 .39861 1.875 0 Mabsadd L
.14095 .39861 0 -2.5 Mabsadd L
.14095 .39861 -1.875 0 Mabsadd L closepath F
.1419 .39884 -1.875 0 Mabsadd m
.1419 .39884 0 2.5 Mabsadd L
.1419 .39884 1.875 0 Mabsadd L
.1419 .39884 0 -2.5 Mabsadd L
.1419 .39884 -1.875 0 Mabsadd L closepath F
.14286 .39945 -1.875 0 Mabsadd m
.14286 .39945 0 2.5 Mabsadd L
.14286 .39945 1.875 0 Mabsadd L
.14286 .39945 0 -2.5 Mabsadd L
.14286 .39945 -1.875 0 Mabsadd L closepath F
.14381 .39948 -1.875 0 Mabsadd m
.14381 .39948 0 2.5 Mabsadd L
.14381 .39948 1.875 0 Mabsadd L
.14381 .39948 0 -2.5 Mabsadd L
.14381 .39948 -1.875 0 Mabsadd L closepath F
.14476 .39958 -1.875 0 Mabsadd m
.14476 .39958 0 2.5 Mabsadd L
.14476 .39958 1.875 0 Mabsadd L
.14476 .39958 0 -2.5 Mabsadd L
.14476 .39958 -1.875 0 Mabsadd L closepath F
.14571 .39958 -1.875 0 Mabsadd m
.14571 .39958 0 2.5 Mabsadd L
.14571 .39958 1.875 0 Mabsadd L
.14571 .39958 0 -2.5 Mabsadd L
.14571 .39958 -1.875 0 Mabsadd L closepath F
.14667 .39961 -1.875 0 Mabsadd m
.14667 .39961 0 2.5 Mabsadd L
.14667 .39961 1.875 0 Mabsadd L
.14667 .39961 0 -2.5 Mabsadd L
.14667 .39961 -1.875 0 Mabsadd L closepath F
.14762 .39964 -1.875 0 Mabsadd m
.14762 .39964 0 2.5 Mabsadd L
.14762 .39964 1.875 0 Mabsadd L
.14762 .39964 0 -2.5 Mabsadd L
.14762 .39964 -1.875 0 Mabsadd L closepath F
.14857 .39964 -1.875 0 Mabsadd m
.14857 .39964 0 2.5 Mabsadd L
.14857 .39964 1.875 0 Mabsadd L
.14857 .39964 0 -2.5 Mabsadd L
.14857 .39964 -1.875 0 Mabsadd L closepath F
.14952 .39965 -1.875 0 Mabsadd m
.14952 .39965 0 2.5 Mabsadd L
.14952 .39965 1.875 0 Mabsadd L
.14952 .39965 0 -2.5 Mabsadd L
.14952 .39965 -1.875 0 Mabsadd L closepath F
.15048 .39966 -1.875 0 Mabsadd m
.15048 .39966 0 2.5 Mabsadd L
.15048 .39966 1.875 0 Mabsadd L
.15048 .39966 0 -2.5 Mabsadd L
.15048 .39966 -1.875 0 Mabsadd L closepath F
.15143 .39966 -1.875 0 Mabsadd m
.15143 .39966 0 2.5 Mabsadd L
.15143 .39966 1.875 0 Mabsadd L
.15143 .39966 0 -2.5 Mabsadd L
.15143 .39966 -1.875 0 Mabsadd L closepath F
.15238 .39966 -1.875 0 Mabsadd m
.15238 .39966 0 2.5 Mabsadd L
.15238 .39966 1.875 0 Mabsadd L
.15238 .39966 0 -2.5 Mabsadd L
.15238 .39966 -1.875 0 Mabsadd L closepath F
.15333 .39966 -1.875 0 Mabsadd m
.15333 .39966 0 2.5 Mabsadd L
.15333 .39966 1.875 0 Mabsadd L
.15333 .39966 0 -2.5 Mabsadd L
.15333 .39966 -1.875 0 Mabsadd L closepath F
.15429 .39966 -1.875 0 Mabsadd m
.15429 .39966 0 2.5 Mabsadd L
.15429 .39966 1.875 0 Mabsadd L
.15429 .39966 0 -2.5 Mabsadd L
.15429 .39966 -1.875 0 Mabsadd L closepath F
.15524 .39966 -1.875 0 Mabsadd m
.15524 .39966 0 2.5 Mabsadd L
.15524 .39966 1.875 0 Mabsadd L
.15524 .39966 0 -2.5 Mabsadd L
.15524 .39966 -1.875 0 Mabsadd L closepath F
.15619 .39966 -1.875 0 Mabsadd m
.15619 .39966 0 2.5 Mabsadd L
.15619 .39966 1.875 0 Mabsadd L
.15619 .39966 0 -2.5 Mabsadd L
.15619 .39966 -1.875 0 Mabsadd L closepath F
.15714 .39966 -1.875 0 Mabsadd m
.15714 .39966 0 2.5 Mabsadd L
.15714 .39966 1.875 0 Mabsadd L
.15714 .39966 0 -2.5 Mabsadd L
.15714 .39966 -1.875 0 Mabsadd L closepath F
.1581 .39966 -1.875 0 Mabsadd m
.1581 .39966 0 2.5 Mabsadd L
.1581 .39966 1.875 0 Mabsadd L
.1581 .39966 0 -2.5 Mabsadd L
.1581 .39966 -1.875 0 Mabsadd L closepath F
.15905 .39966 -1.875 0 Mabsadd m
.15905 .39966 0 2.5 Mabsadd L
.15905 .39966 1.875 0 Mabsadd L
.15905 .39966 0 -2.5 Mabsadd L
.15905 .39966 -1.875 0 Mabsadd L closepath F
.16 .39966 -1.875 0 Mabsadd m
.16 .39966 0 2.5 Mabsadd L
.16 .39966 1.875 0 Mabsadd L
.16 .39966 0 -2.5 Mabsadd L
.16 .39966 -1.875 0 Mabsadd L closepath F
.16095 .39966 -1.875 0 Mabsadd m
.16095 .39966 0 2.5 Mabsadd L
.16095 .39966 1.875 0 Mabsadd L
.16095 .39966 0 -2.5 Mabsadd L
.16095 .39966 -1.875 0 Mabsadd L closepath F
.1619 .39965 -1.875 0 Mabsadd m
.1619 .39965 0 2.5 Mabsadd L
.1619 .39965 1.875 0 Mabsadd L
.1619 .39965 0 -2.5 Mabsadd L
.1619 .39965 -1.875 0 Mabsadd L closepath F
.16286 .39965 -1.875 0 Mabsadd m
.16286 .39965 0 2.5 Mabsadd L
.16286 .39965 1.875 0 Mabsadd L
.16286 .39965 0 -2.5 Mabsadd L
.16286 .39965 -1.875 0 Mabsadd L closepath F
.16381 .39965 -1.875 0 Mabsadd m
.16381 .39965 0 2.5 Mabsadd L
.16381 .39965 1.875 0 Mabsadd L
.16381 .39965 0 -2.5 Mabsadd L
.16381 .39965 -1.875 0 Mabsadd L closepath F
.16476 .39965 -1.875 0 Mabsadd m
.16476 .39965 0 2.5 Mabsadd L
.16476 .39965 1.875 0 Mabsadd L
.16476 .39965 0 -2.5 Mabsadd L
.16476 .39965 -1.875 0 Mabsadd L closepath F
.16571 .39965 -1.875 0 Mabsadd m
.16571 .39965 0 2.5 Mabsadd L
.16571 .39965 1.875 0 Mabsadd L
.16571 .39965 0 -2.5 Mabsadd L
.16571 .39965 -1.875 0 Mabsadd L closepath F
.16667 .39965 -1.875 0 Mabsadd m
.16667 .39965 0 2.5 Mabsadd L
.16667 .39965 1.875 0 Mabsadd L
.16667 .39965 0 -2.5 Mabsadd L
.16667 .39965 -1.875 0 Mabsadd L closepath F
.16762 .39964 -1.875 0 Mabsadd m
.16762 .39964 0 2.5 Mabsadd L
.16762 .39964 1.875 0 Mabsadd L
.16762 .39964 0 -2.5 Mabsadd L
.16762 .39964 -1.875 0 Mabsadd L closepath F
.16857 .39963 -1.875 0 Mabsadd m
.16857 .39963 0 2.5 Mabsadd L
.16857 .39963 1.875 0 Mabsadd L
.16857 .39963 0 -2.5 Mabsadd L
.16857 .39963 -1.875 0 Mabsadd L closepath F
.16952 .39963 -1.875 0 Mabsadd m
.16952 .39963 0 2.5 Mabsadd L
.16952 .39963 1.875 0 Mabsadd L
.16952 .39963 0 -2.5 Mabsadd L
.16952 .39963 -1.875 0 Mabsadd L closepath F
.17048 .39962 -1.875 0 Mabsadd m
.17048 .39962 0 2.5 Mabsadd L
.17048 .39962 1.875 0 Mabsadd L
.17048 .39962 0 -2.5 Mabsadd L
.17048 .39962 -1.875 0 Mabsadd L closepath F
.17143 .39961 -1.875 0 Mabsadd m
.17143 .39961 0 2.5 Mabsadd L
.17143 .39961 1.875 0 Mabsadd L
.17143 .39961 0 -2.5 Mabsadd L
.17143 .39961 -1.875 0 Mabsadd L closepath F
.17238 .39954 -1.875 0 Mabsadd m
.17238 .39954 0 2.5 Mabsadd L
.17238 .39954 1.875 0 Mabsadd L
.17238 .39954 0 -2.5 Mabsadd L
.17238 .39954 -1.875 0 Mabsadd L closepath F
.17333 .39951 -1.875 0 Mabsadd m
.17333 .39951 0 2.5 Mabsadd L
.17333 .39951 1.875 0 Mabsadd L
.17333 .39951 0 -2.5 Mabsadd L
.17333 .39951 -1.875 0 Mabsadd L closepath F
.17429 .39911 -1.875 0 Mabsadd m
.17429 .39911 0 2.5 Mabsadd L
.17429 .39911 1.875 0 Mabsadd L
.17429 .39911 0 -2.5 Mabsadd L
.17429 .39911 -1.875 0 Mabsadd L closepath F
.17524 .39883 -1.875 0 Mabsadd m
.17524 .39883 0 2.5 Mabsadd L
.17524 .39883 1.875 0 Mabsadd L
.17524 .39883 0 -2.5 Mabsadd L
.17524 .39883 -1.875 0 Mabsadd L closepath F
.17619 .3948 -1.875 0 Mabsadd m
.17619 .3948 0 2.5 Mabsadd L
.17619 .3948 1.875 0 Mabsadd L
.17619 .3948 0 -2.5 Mabsadd L
.17619 .3948 -1.875 0 Mabsadd L closepath F
.17714 .39086 -1.875 0 Mabsadd m
.17714 .39086 0 2.5 Mabsadd L
.17714 .39086 1.875 0 Mabsadd L
.17714 .39086 0 -2.5 Mabsadd L
.17714 .39086 -1.875 0 Mabsadd L closepath F
.1781 .37193 -1.875 0 Mabsadd m
.1781 .37193 0 2.5 Mabsadd L
.1781 .37193 1.875 0 Mabsadd L
.1781 .37193 0 -2.5 Mabsadd L
.1781 .37193 -1.875 0 Mabsadd L closepath F
.17905 .37191 -1.875 0 Mabsadd m
.17905 .37191 0 2.5 Mabsadd L
.17905 .37191 1.875 0 Mabsadd L
.17905 .37191 0 -2.5 Mabsadd L
.17905 .37191 -1.875 0 Mabsadd L closepath F
.18 .37263 -1.875 0 Mabsadd m
.18 .37263 0 2.5 Mabsadd L
.18 .37263 1.875 0 Mabsadd L
.18 .37263 0 -2.5 Mabsadd L
.18 .37263 -1.875 0 Mabsadd L closepath F
.18095 .38061 -1.875 0 Mabsadd m
.18095 .38061 0 2.5 Mabsadd L
.18095 .38061 1.875 0 Mabsadd L
.18095 .38061 0 -2.5 Mabsadd L
.18095 .38061 -1.875 0 Mabsadd L closepath F
.1819 .38912 -1.875 0 Mabsadd m
.1819 .38912 0 2.5 Mabsadd L
.1819 .38912 1.875 0 Mabsadd L
.1819 .38912 0 -2.5 Mabsadd L
.1819 .38912 -1.875 0 Mabsadd L closepath F
.18286 .36393 -1.875 0 Mabsadd m
.18286 .36393 0 2.5 Mabsadd L
.18286 .36393 1.875 0 Mabsadd L
.18286 .36393 0 -2.5 Mabsadd L
.18286 .36393 -1.875 0 Mabsadd L closepath F
.18381 .3164 -1.875 0 Mabsadd m
.18381 .3164 0 2.5 Mabsadd L
.18381 .3164 1.875 0 Mabsadd L
.18381 .3164 0 -2.5 Mabsadd L
.18381 .3164 -1.875 0 Mabsadd L closepath F
.18476 .41396 -1.875 0 Mabsadd m
.18476 .41396 0 2.5 Mabsadd L
.18476 .41396 1.875 0 Mabsadd L
.18476 .41396 0 -2.5 Mabsadd L
.18476 .41396 -1.875 0 Mabsadd L closepath F
.18571 .40878 -1.875 0 Mabsadd m
.18571 .40878 0 2.5 Mabsadd L
.18571 .40878 1.875 0 Mabsadd L
.18571 .40878 0 -2.5 Mabsadd L
.18571 .40878 -1.875 0 Mabsadd L closepath F
.18667 .43743 -1.875 0 Mabsadd m
.18667 .43743 0 2.5 Mabsadd L
.18667 .43743 1.875 0 Mabsadd L
.18667 .43743 0 -2.5 Mabsadd L
.18667 .43743 -1.875 0 Mabsadd L closepath F
.18762 .43471 -1.875 0 Mabsadd m
.18762 .43471 0 2.5 Mabsadd L
.18762 .43471 1.875 0 Mabsadd L
.18762 .43471 0 -2.5 Mabsadd L
.18762 .43471 -1.875 0 Mabsadd L closepath F
.18857 .43716 -1.875 0 Mabsadd m
.18857 .43716 0 2.5 Mabsadd L
.18857 .43716 1.875 0 Mabsadd L
.18857 .43716 0 -2.5 Mabsadd L
.18857 .43716 -1.875 0 Mabsadd L closepath F
.18952 .45199 -1.875 0 Mabsadd m
.18952 .45199 0 2.5 Mabsadd L
.18952 .45199 1.875 0 Mabsadd L
.18952 .45199 0 -2.5 Mabsadd L
.18952 .45199 -1.875 0 Mabsadd L closepath F
.19048 .45204 -1.875 0 Mabsadd m
.19048 .45204 0 2.5 Mabsadd L
.19048 .45204 1.875 0 Mabsadd L
.19048 .45204 0 -2.5 Mabsadd L
.19048 .45204 -1.875 0 Mabsadd L closepath F
.19143 .45219 -1.875 0 Mabsadd m
.19143 .45219 0 2.5 Mabsadd L
.19143 .45219 1.875 0 Mabsadd L
.19143 .45219 0 -2.5 Mabsadd L
.19143 .45219 -1.875 0 Mabsadd L closepath F
.19238 .45235 -1.875 0 Mabsadd m
.19238 .45235 0 2.5 Mabsadd L
.19238 .45235 1.875 0 Mabsadd L
.19238 .45235 0 -2.5 Mabsadd L
.19238 .45235 -1.875 0 Mabsadd L closepath F
.19333 .45245 -1.875 0 Mabsadd m
.19333 .45245 0 2.5 Mabsadd L
.19333 .45245 1.875 0 Mabsadd L
.19333 .45245 0 -2.5 Mabsadd L
.19333 .45245 -1.875 0 Mabsadd L closepath F
.19429 .4525 -1.875 0 Mabsadd m
.19429 .4525 0 2.5 Mabsadd L
.19429 .4525 1.875 0 Mabsadd L
.19429 .4525 0 -2.5 Mabsadd L
.19429 .4525 -1.875 0 Mabsadd L closepath F
.19524 .45276 -1.875 0 Mabsadd m
.19524 .45276 0 2.5 Mabsadd L
.19524 .45276 1.875 0 Mabsadd L
.19524 .45276 0 -2.5 Mabsadd L
.19524 .45276 -1.875 0 Mabsadd L closepath F
.19619 .45346 -1.875 0 Mabsadd m
.19619 .45346 0 2.5 Mabsadd L
.19619 .45346 1.875 0 Mabsadd L
.19619 .45346 0 -2.5 Mabsadd L
.19619 .45346 -1.875 0 Mabsadd L closepath F
.19714 .45349 -1.875 0 Mabsadd m
.19714 .45349 0 2.5 Mabsadd L
.19714 .45349 1.875 0 Mabsadd L
.19714 .45349 0 -2.5 Mabsadd L
.19714 .45349 -1.875 0 Mabsadd L closepath F
.1981 .4536 -1.875 0 Mabsadd m
.1981 .4536 0 2.5 Mabsadd L
.1981 .4536 1.875 0 Mabsadd L
.1981 .4536 0 -2.5 Mabsadd L
.1981 .4536 -1.875 0 Mabsadd L closepath F
.19905 .45361 -1.875 0 Mabsadd m
.19905 .45361 0 2.5 Mabsadd L
.19905 .45361 1.875 0 Mabsadd L
.19905 .45361 0 -2.5 Mabsadd L
.19905 .45361 -1.875 0 Mabsadd L closepath F
.2 .45364 -1.875 0 Mabsadd m
.2 .45364 0 2.5 Mabsadd L
.2 .45364 1.875 0 Mabsadd L
.2 .45364 0 -2.5 Mabsadd L
.2 .45364 -1.875 0 Mabsadd L closepath F
.20095 .45367 -1.875 0 Mabsadd m
.20095 .45367 0 2.5 Mabsadd L
.20095 .45367 1.875 0 Mabsadd L
.20095 .45367 0 -2.5 Mabsadd L
.20095 .45367 -1.875 0 Mabsadd L closepath F
.2019 .45368 -1.875 0 Mabsadd m
.2019 .45368 0 2.5 Mabsadd L
.2019 .45368 1.875 0 Mabsadd L
.2019 .45368 0 -2.5 Mabsadd L
.2019 .45368 -1.875 0 Mabsadd L closepath F
.20286 .45368 -1.875 0 Mabsadd m
.20286 .45368 0 2.5 Mabsadd L
.20286 .45368 1.875 0 Mabsadd L
.20286 .45368 0 -2.5 Mabsadd L
.20286 .45368 -1.875 0 Mabsadd L closepath F
.20381 .45369 -1.875 0 Mabsadd m
.20381 .45369 0 2.5 Mabsadd L
.20381 .45369 1.875 0 Mabsadd L
.20381 .45369 0 -2.5 Mabsadd L
.20381 .45369 -1.875 0 Mabsadd L closepath F
.20476 .45369 -1.875 0 Mabsadd m
.20476 .45369 0 2.5 Mabsadd L
.20476 .45369 1.875 0 Mabsadd L
.20476 .45369 0 -2.5 Mabsadd L
.20476 .45369 -1.875 0 Mabsadd L closepath F
.20571 .45369 -1.875 0 Mabsadd m
.20571 .45369 0 2.5 Mabsadd L
.20571 .45369 1.875 0 Mabsadd L
.20571 .45369 0 -2.5 Mabsadd L
.20571 .45369 -1.875 0 Mabsadd L closepath F
.20667 .45369 -1.875 0 Mabsadd m
.20667 .45369 0 2.5 Mabsadd L
.20667 .45369 1.875 0 Mabsadd L
.20667 .45369 0 -2.5 Mabsadd L
.20667 .45369 -1.875 0 Mabsadd L closepath F
.20762 .45369 -1.875 0 Mabsadd m
.20762 .45369 0 2.5 Mabsadd L
.20762 .45369 1.875 0 Mabsadd L
.20762 .45369 0 -2.5 Mabsadd L
.20762 .45369 -1.875 0 Mabsadd L closepath F
.20857 .45369 -1.875 0 Mabsadd m
.20857 .45369 0 2.5 Mabsadd L
.20857 .45369 1.875 0 Mabsadd L
.20857 .45369 0 -2.5 Mabsadd L
.20857 .45369 -1.875 0 Mabsadd L closepath F
.20952 .45369 -1.875 0 Mabsadd m
.20952 .45369 0 2.5 Mabsadd L
.20952 .45369 1.875 0 Mabsadd L
.20952 .45369 0 -2.5 Mabsadd L
.20952 .45369 -1.875 0 Mabsadd L closepath F
.21048 .45369 -1.875 0 Mabsadd m
.21048 .45369 0 2.5 Mabsadd L
.21048 .45369 1.875 0 Mabsadd L
.21048 .45369 0 -2.5 Mabsadd L
.21048 .45369 -1.875 0 Mabsadd L closepath F
.21143 .45369 -1.875 0 Mabsadd m
.21143 .45369 0 2.5 Mabsadd L
.21143 .45369 1.875 0 Mabsadd L
.21143 .45369 0 -2.5 Mabsadd L
.21143 .45369 -1.875 0 Mabsadd L closepath F
.21238 .45369 -1.875 0 Mabsadd m
.21238 .45369 0 2.5 Mabsadd L
.21238 .45369 1.875 0 Mabsadd L
.21238 .45369 0 -2.5 Mabsadd L
.21238 .45369 -1.875 0 Mabsadd L closepath F
.21333 .45369 -1.875 0 Mabsadd m
.21333 .45369 0 2.5 Mabsadd L
.21333 .45369 1.875 0 Mabsadd L
.21333 .45369 0 -2.5 Mabsadd L
.21333 .45369 -1.875 0 Mabsadd L closepath F
.21429 .45369 -1.875 0 Mabsadd m
.21429 .45369 0 2.5 Mabsadd L
.21429 .45369 1.875 0 Mabsadd L
.21429 .45369 0 -2.5 Mabsadd L
.21429 .45369 -1.875 0 Mabsadd L closepath F
.21524 .45369 -1.875 0 Mabsadd m
.21524 .45369 0 2.5 Mabsadd L
.21524 .45369 1.875 0 Mabsadd L
.21524 .45369 0 -2.5 Mabsadd L
.21524 .45369 -1.875 0 Mabsadd L closepath F
.21619 .45369 -1.875 0 Mabsadd m
.21619 .45369 0 2.5 Mabsadd L
.21619 .45369 1.875 0 Mabsadd L
.21619 .45369 0 -2.5 Mabsadd L
.21619 .45369 -1.875 0 Mabsadd L closepath F
.21714 .45369 -1.875 0 Mabsadd m
.21714 .45369 0 2.5 Mabsadd L
.21714 .45369 1.875 0 Mabsadd L
.21714 .45369 0 -2.5 Mabsadd L
.21714 .45369 -1.875 0 Mabsadd L closepath F
.2181 .45369 -1.875 0 Mabsadd m
.2181 .45369 0 2.5 Mabsadd L
.2181 .45369 1.875 0 Mabsadd L
.2181 .45369 0 -2.5 Mabsadd L
.2181 .45369 -1.875 0 Mabsadd L closepath F
.21905 .45368 -1.875 0 Mabsadd m
.21905 .45368 0 2.5 Mabsadd L
.21905 .45368 1.875 0 Mabsadd L
.21905 .45368 0 -2.5 Mabsadd L
.21905 .45368 -1.875 0 Mabsadd L closepath F
.22 .45368 -1.875 0 Mabsadd m
.22 .45368 0 2.5 Mabsadd L
.22 .45368 1.875 0 Mabsadd L
.22 .45368 0 -2.5 Mabsadd L
.22 .45368 -1.875 0 Mabsadd L closepath F
.22095 .45368 -1.875 0 Mabsadd m
.22095 .45368 0 2.5 Mabsadd L
.22095 .45368 1.875 0 Mabsadd L
.22095 .45368 0 -2.5 Mabsadd L
.22095 .45368 -1.875 0 Mabsadd L closepath F
.2219 .45367 -1.875 0 Mabsadd m
.2219 .45367 0 2.5 Mabsadd L
.2219 .45367 1.875 0 Mabsadd L
.2219 .45367 0 -2.5 Mabsadd L
.2219 .45367 -1.875 0 Mabsadd L closepath F
.22286 .45367 -1.875 0 Mabsadd m
.22286 .45367 0 2.5 Mabsadd L
.22286 .45367 1.875 0 Mabsadd L
.22286 .45367 0 -2.5 Mabsadd L
.22286 .45367 -1.875 0 Mabsadd L closepath F
.22381 .45365 -1.875 0 Mabsadd m
.22381 .45365 0 2.5 Mabsadd L
.22381 .45365 1.875 0 Mabsadd L
.22381 .45365 0 -2.5 Mabsadd L
.22381 .45365 -1.875 0 Mabsadd L closepath F
.22476 .45359 -1.875 0 Mabsadd m
.22476 .45359 0 2.5 Mabsadd L
.22476 .45359 1.875 0 Mabsadd L
.22476 .45359 0 -2.5 Mabsadd L
.22476 .45359 -1.875 0 Mabsadd L closepath F
.22571 .45358 -1.875 0 Mabsadd m
.22571 .45358 0 2.5 Mabsadd L
.22571 .45358 1.875 0 Mabsadd L
.22571 .45358 0 -2.5 Mabsadd L
.22571 .45358 -1.875 0 Mabsadd L closepath F
.22667 .45345 -1.875 0 Mabsadd m
.22667 .45345 0 2.5 Mabsadd L
.22667 .45345 1.875 0 Mabsadd L
.22667 .45345 0 -2.5 Mabsadd L
.22667 .45345 -1.875 0 Mabsadd L closepath F
.22762 .45336 -1.875 0 Mabsadd m
.22762 .45336 0 2.5 Mabsadd L
.22762 .45336 1.875 0 Mabsadd L
.22762 .45336 0 -2.5 Mabsadd L
.22762 .45336 -1.875 0 Mabsadd L closepath F
.22857 .45234 -1.875 0 Mabsadd m
.22857 .45234 0 2.5 Mabsadd L
.22857 .45234 1.875 0 Mabsadd L
.22857 .45234 0 -2.5 Mabsadd L
.22857 .45234 -1.875 0 Mabsadd L closepath F
.22952 .45152 -1.875 0 Mabsadd m
.22952 .45152 0 2.5 Mabsadd L
.22952 .45152 1.875 0 Mabsadd L
.22952 .45152 0 -2.5 Mabsadd L
.22952 .45152 -1.875 0 Mabsadd L closepath F
.23048 .43833 -1.875 0 Mabsadd m
.23048 .43833 0 2.5 Mabsadd L
.23048 .43833 1.875 0 Mabsadd L
.23048 .43833 0 -2.5 Mabsadd L
.23048 .43833 -1.875 0 Mabsadd L closepath F
.23143 .42319 -1.875 0 Mabsadd m
.23143 .42319 0 2.5 Mabsadd L
.23143 .42319 1.875 0 Mabsadd L
.23143 .42319 0 -2.5 Mabsadd L
.23143 .42319 -1.875 0 Mabsadd L closepath F
.23238 .33965 -1.875 0 Mabsadd m
.23238 .33965 0 2.5 Mabsadd L
.23238 .33965 1.875 0 Mabsadd L
.23238 .33965 0 -2.5 Mabsadd L
.23238 .33965 -1.875 0 Mabsadd L closepath F
.23333 .33969 -1.875 0 Mabsadd m
.23333 .33969 0 2.5 Mabsadd L
.23333 .33969 1.875 0 Mabsadd L
.23333 .33969 0 -2.5 Mabsadd L
.23333 .33969 -1.875 0 Mabsadd L closepath F
.23429 .34421 -1.875 0 Mabsadd m
.23429 .34421 0 2.5 Mabsadd L
.23429 .34421 1.875 0 Mabsadd L
.23429 .34421 0 -2.5 Mabsadd L
.23429 .34421 -1.875 0 Mabsadd L closepath F
.23524 .34998 -1.875 0 Mabsadd m
.23524 .34998 0 2.5 Mabsadd L
.23524 .34998 1.875 0 Mabsadd L
.23524 .34998 0 -2.5 Mabsadd L
.23524 .34998 -1.875 0 Mabsadd L closepath F
.23619 .32069 -1.875 0 Mabsadd m
.23619 .32069 0 2.5 Mabsadd L
.23619 .32069 1.875 0 Mabsadd L
.23619 .32069 0 -2.5 Mabsadd L
.23619 .32069 -1.875 0 Mabsadd L closepath F
.23714 .31263 -1.875 0 Mabsadd m
.23714 .31263 0 2.5 Mabsadd L
.23714 .31263 1.875 0 Mabsadd L
.23714 .31263 0 -2.5 Mabsadd L
.23714 .31263 -1.875 0 Mabsadd L closepath F
.2381 .317 -1.875 0 Mabsadd m
.2381 .317 0 2.5 Mabsadd L
.2381 .317 1.875 0 Mabsadd L
.2381 .317 0 -2.5 Mabsadd L
.2381 .317 -1.875 0 Mabsadd L closepath F
.23905 .34294 -1.875 0 Mabsadd m
.23905 .34294 0 2.5 Mabsadd L
.23905 .34294 1.875 0 Mabsadd L
.23905 .34294 0 -2.5 Mabsadd L
.23905 .34294 -1.875 0 Mabsadd L closepath F
.24 .37005 -1.875 0 Mabsadd m
.24 .37005 0 2.5 Mabsadd L
.24 .37005 1.875 0 Mabsadd L
.24 .37005 0 -2.5 Mabsadd L
.24 .37005 -1.875 0 Mabsadd L closepath F
.24095 .36518 -1.875 0 Mabsadd m
.24095 .36518 0 2.5 Mabsadd L
.24095 .36518 1.875 0 Mabsadd L
.24095 .36518 0 -2.5 Mabsadd L
.24095 .36518 -1.875 0 Mabsadd L closepath F
.2419 .36852 -1.875 0 Mabsadd m
.2419 .36852 0 2.5 Mabsadd L
.2419 .36852 1.875 0 Mabsadd L
.2419 .36852 0 -2.5 Mabsadd L
.2419 .36852 -1.875 0 Mabsadd L closepath F
.24286 .38948 -1.875 0 Mabsadd m
.24286 .38948 0 2.5 Mabsadd L
.24286 .38948 1.875 0 Mabsadd L
.24286 .38948 0 -2.5 Mabsadd L
.24286 .38948 -1.875 0 Mabsadd L closepath F
.24381 .38954 -1.875 0 Mabsadd m
.24381 .38954 0 2.5 Mabsadd L
.24381 .38954 1.875 0 Mabsadd L
.24381 .38954 0 -2.5 Mabsadd L
.24381 .38954 -1.875 0 Mabsadd L closepath F
.24476 .38974 -1.875 0 Mabsadd m
.24476 .38974 0 2.5 Mabsadd L
.24476 .38974 1.875 0 Mabsadd L
.24476 .38974 0 -2.5 Mabsadd L
.24476 .38974 -1.875 0 Mabsadd L closepath F
.24571 .38994 -1.875 0 Mabsadd m
.24571 .38994 0 2.5 Mabsadd L
.24571 .38994 1.875 0 Mabsadd L
.24571 .38994 0 -2.5 Mabsadd L
.24571 .38994 -1.875 0 Mabsadd L closepath F
.24667 .39007 -1.875 0 Mabsadd m
.24667 .39007 0 2.5 Mabsadd L
.24667 .39007 1.875 0 Mabsadd L
.24667 .39007 0 -2.5 Mabsadd L
.24667 .39007 -1.875 0 Mabsadd L closepath F
.24762 .39013 -1.875 0 Mabsadd m
.24762 .39013 0 2.5 Mabsadd L
.24762 .39013 1.875 0 Mabsadd L
.24762 .39013 0 -2.5 Mabsadd L
.24762 .39013 -1.875 0 Mabsadd L closepath F
.24857 .39043 -1.875 0 Mabsadd m
.24857 .39043 0 2.5 Mabsadd L
.24857 .39043 1.875 0 Mabsadd L
.24857 .39043 0 -2.5 Mabsadd L
.24857 .39043 -1.875 0 Mabsadd L closepath F
.24952 .39123 -1.875 0 Mabsadd m
.24952 .39123 0 2.5 Mabsadd L
.24952 .39123 1.875 0 Mabsadd L
.24952 .39123 0 -2.5 Mabsadd L
.24952 .39123 -1.875 0 Mabsadd L closepath F
.25048 .39127 -1.875 0 Mabsadd m
.25048 .39127 0 2.5 Mabsadd L
.25048 .39127 1.875 0 Mabsadd L
.25048 .39127 0 -2.5 Mabsadd L
.25048 .39127 -1.875 0 Mabsadd L closepath F
.25143 .39139 -1.875 0 Mabsadd m
.25143 .39139 0 2.5 Mabsadd L
.25143 .39139 1.875 0 Mabsadd L
.25143 .39139 0 -2.5 Mabsadd L
.25143 .39139 -1.875 0 Mabsadd L closepath F
.25238 .3914 -1.875 0 Mabsadd m
.25238 .3914 0 2.5 Mabsadd L
.25238 .3914 1.875 0 Mabsadd L
.25238 .3914 0 -2.5 Mabsadd L
.25238 .3914 -1.875 0 Mabsadd L closepath F
.25333 .39143 -1.875 0 Mabsadd m
.25333 .39143 0 2.5 Mabsadd L
.25333 .39143 1.875 0 Mabsadd L
.25333 .39143 0 -2.5 Mabsadd L
.25333 .39143 -1.875 0 Mabsadd L closepath F
.25429 .39147 -1.875 0 Mabsadd m
.25429 .39147 0 2.5 Mabsadd L
.25429 .39147 1.875 0 Mabsadd L
.25429 .39147 0 -2.5 Mabsadd L
.25429 .39147 -1.875 0 Mabsadd L closepath F
.25524 .39147 -1.875 0 Mabsadd m
.25524 .39147 0 2.5 Mabsadd L
.25524 .39147 1.875 0 Mabsadd L
.25524 .39147 0 -2.5 Mabsadd L
.25524 .39147 -1.875 0 Mabsadd L closepath F
.25619 .39147 -1.875 0 Mabsadd m
.25619 .39147 0 2.5 Mabsadd L
.25619 .39147 1.875 0 Mabsadd L
.25619 .39147 0 -2.5 Mabsadd L
.25619 .39147 -1.875 0 Mabsadd L closepath F
.25714 .39148 -1.875 0 Mabsadd m
.25714 .39148 0 2.5 Mabsadd L
.25714 .39148 1.875 0 Mabsadd L
.25714 .39148 0 -2.5 Mabsadd L
.25714 .39148 -1.875 0 Mabsadd L closepath F
.2581 .39148 -1.875 0 Mabsadd m
.2581 .39148 0 2.5 Mabsadd L
.2581 .39148 1.875 0 Mabsadd L
.2581 .39148 0 -2.5 Mabsadd L
.2581 .39148 -1.875 0 Mabsadd L closepath F
.25905 .39148 -1.875 0 Mabsadd m
.25905 .39148 0 2.5 Mabsadd L
.25905 .39148 1.875 0 Mabsadd L
.25905 .39148 0 -2.5 Mabsadd L
.25905 .39148 -1.875 0 Mabsadd L closepath F
.26 .39148 -1.875 0 Mabsadd m
.26 .39148 0 2.5 Mabsadd L
.26 .39148 1.875 0 Mabsadd L
.26 .39148 0 -2.5 Mabsadd L
.26 .39148 -1.875 0 Mabsadd L closepath F
.26095 .39149 -1.875 0 Mabsadd m
.26095 .39149 0 2.5 Mabsadd L
.26095 .39149 1.875 0 Mabsadd L
.26095 .39149 0 -2.5 Mabsadd L
.26095 .39149 -1.875 0 Mabsadd L closepath F
.2619 .39149 -1.875 0 Mabsadd m
.2619 .39149 0 2.5 Mabsadd L
.2619 .39149 1.875 0 Mabsadd L
.2619 .39149 0 -2.5 Mabsadd L
.2619 .39149 -1.875 0 Mabsadd L closepath F
.26286 .39149 -1.875 0 Mabsadd m
.26286 .39149 0 2.5 Mabsadd L
.26286 .39149 1.875 0 Mabsadd L
.26286 .39149 0 -2.5 Mabsadd L
.26286 .39149 -1.875 0 Mabsadd L closepath F
.26381 .39149 -1.875 0 Mabsadd m
.26381 .39149 0 2.5 Mabsadd L
.26381 .39149 1.875 0 Mabsadd L
.26381 .39149 0 -2.5 Mabsadd L
.26381 .39149 -1.875 0 Mabsadd L closepath F
.26476 .39149 -1.875 0 Mabsadd m
.26476 .39149 0 2.5 Mabsadd L
.26476 .39149 1.875 0 Mabsadd L
.26476 .39149 0 -2.5 Mabsadd L
.26476 .39149 -1.875 0 Mabsadd L closepath F
.26571 .39149 -1.875 0 Mabsadd m
.26571 .39149 0 2.5 Mabsadd L
.26571 .39149 1.875 0 Mabsadd L
.26571 .39149 0 -2.5 Mabsadd L
.26571 .39149 -1.875 0 Mabsadd L closepath F
.26667 .39149 -1.875 0 Mabsadd m
.26667 .39149 0 2.5 Mabsadd L
.26667 .39149 1.875 0 Mabsadd L
.26667 .39149 0 -2.5 Mabsadd L
.26667 .39149 -1.875 0 Mabsadd L closepath F
.26762 .39149 -1.875 0 Mabsadd m
.26762 .39149 0 2.5 Mabsadd L
.26762 .39149 1.875 0 Mabsadd L
.26762 .39149 0 -2.5 Mabsadd L
.26762 .39149 -1.875 0 Mabsadd L closepath F
.26857 .39149 -1.875 0 Mabsadd m
.26857 .39149 0 2.5 Mabsadd L
.26857 .39149 1.875 0 Mabsadd L
.26857 .39149 0 -2.5 Mabsadd L
.26857 .39149 -1.875 0 Mabsadd L closepath F
.26952 .39149 -1.875 0 Mabsadd m
.26952 .39149 0 2.5 Mabsadd L
.26952 .39149 1.875 0 Mabsadd L
.26952 .39149 0 -2.5 Mabsadd L
.26952 .39149 -1.875 0 Mabsadd L closepath F
.27048 .39148 -1.875 0 Mabsadd m
.27048 .39148 0 2.5 Mabsadd L
.27048 .39148 1.875 0 Mabsadd L
.27048 .39148 0 -2.5 Mabsadd L
.27048 .39148 -1.875 0 Mabsadd L closepath F
.27143 .39148 -1.875 0 Mabsadd m
.27143 .39148 0 2.5 Mabsadd L
.27143 .39148 1.875 0 Mabsadd L
.27143 .39148 0 -2.5 Mabsadd L
.27143 .39148 -1.875 0 Mabsadd L closepath F
.27238 .39148 -1.875 0 Mabsadd m
.27238 .39148 0 2.5 Mabsadd L
.27238 .39148 1.875 0 Mabsadd L
.27238 .39148 0 -2.5 Mabsadd L
.27238 .39148 -1.875 0 Mabsadd L closepath F
.27333 .39148 -1.875 0 Mabsadd m
.27333 .39148 0 2.5 Mabsadd L
.27333 .39148 1.875 0 Mabsadd L
.27333 .39148 0 -2.5 Mabsadd L
.27333 .39148 -1.875 0 Mabsadd L closepath F
.27429 .39147 -1.875 0 Mabsadd m
.27429 .39147 0 2.5 Mabsadd L
.27429 .39147 1.875 0 Mabsadd L
.27429 .39147 0 -2.5 Mabsadd L
.27429 .39147 -1.875 0 Mabsadd L closepath F
.27524 .39147 -1.875 0 Mabsadd m
.27524 .39147 0 2.5 Mabsadd L
.27524 .39147 1.875 0 Mabsadd L
.27524 .39147 0 -2.5 Mabsadd L
.27524 .39147 -1.875 0 Mabsadd L closepath F
.27619 .39147 -1.875 0 Mabsadd m
.27619 .39147 0 2.5 Mabsadd L
.27619 .39147 1.875 0 Mabsadd L
.27619 .39147 0 -2.5 Mabsadd L
.27619 .39147 -1.875 0 Mabsadd L closepath F
.27714 .39144 -1.875 0 Mabsadd m
.27714 .39144 0 2.5 Mabsadd L
.27714 .39144 1.875 0 Mabsadd L
.27714 .39144 0 -2.5 Mabsadd L
.27714 .39144 -1.875 0 Mabsadd L closepath F
.2781 .39143 -1.875 0 Mabsadd m
.2781 .39143 0 2.5 Mabsadd L
.2781 .39143 1.875 0 Mabsadd L
.2781 .39143 0 -2.5 Mabsadd L
.2781 .39143 -1.875 0 Mabsadd L closepath F
.27905 .39138 -1.875 0 Mabsadd m
.27905 .39138 0 2.5 Mabsadd L
.27905 .39138 1.875 0 Mabsadd L
.27905 .39138 0 -2.5 Mabsadd L
.27905 .39138 -1.875 0 Mabsadd L closepath F
.28 .39114 -1.875 0 Mabsadd m
.28 .39114 0 2.5 Mabsadd L
.28 .39114 1.875 0 Mabsadd L
.28 .39114 0 -2.5 Mabsadd L
.28 .39114 -1.875 0 Mabsadd L closepath F
.28095 .39107 -1.875 0 Mabsadd m
.28095 .39107 0 2.5 Mabsadd L
.28095 .39107 1.875 0 Mabsadd L
.28095 .39107 0 -2.5 Mabsadd L
.28095 .39107 -1.875 0 Mabsadd L closepath F
.2819 .39018 -1.875 0 Mabsadd m
.2819 .39018 0 2.5 Mabsadd L
.2819 .39018 1.875 0 Mabsadd L
.2819 .39018 0 -2.5 Mabsadd L
.2819 .39018 -1.875 0 Mabsadd L closepath F
.28286 .38949 -1.875 0 Mabsadd m
.28286 .38949 0 2.5 Mabsadd L
.28286 .38949 1.875 0 Mabsadd L
.28286 .38949 0 -2.5 Mabsadd L
.28286 .38949 -1.875 0 Mabsadd L closepath F
.28381 .37851 -1.875 0 Mabsadd m
.28381 .37851 0 2.5 Mabsadd L
.28381 .37851 1.875 0 Mabsadd L
.28381 .37851 0 -2.5 Mabsadd L
.28381 .37851 -1.875 0 Mabsadd L closepath F
.28476 .36621 -1.875 0 Mabsadd m
.28476 .36621 0 2.5 Mabsadd L
.28476 .36621 1.875 0 Mabsadd L
.28476 .36621 0 -2.5 Mabsadd L
.28476 .36621 -1.875 0 Mabsadd L closepath F
.28571 .29972 -1.875 0 Mabsadd m
.28571 .29972 0 2.5 Mabsadd L
.28571 .29972 1.875 0 Mabsadd L
.28571 .29972 0 -2.5 Mabsadd L
.28571 .29972 -1.875 0 Mabsadd L closepath F
.28667 .29974 -1.875 0 Mabsadd m
.28667 .29974 0 2.5 Mabsadd L
.28667 .29974 1.875 0 Mabsadd L
.28667 .29974 0 -2.5 Mabsadd L
.28667 .29974 -1.875 0 Mabsadd L closepath F
.28762 .30319 -1.875 0 Mabsadd m
.28762 .30319 0 2.5 Mabsadd L
.28762 .30319 1.875 0 Mabsadd L
.28762 .30319 0 -2.5 Mabsadd L
.28762 .30319 -1.875 0 Mabsadd L closepath F
.28857 .30727 -1.875 0 Mabsadd m
.28857 .30727 0 2.5 Mabsadd L
.28857 .30727 1.875 0 Mabsadd L
.28857 .30727 0 -2.5 Mabsadd L
.28857 .30727 -1.875 0 Mabsadd L closepath F
.28952 .27755 -1.875 0 Mabsadd m
.28952 .27755 0 2.5 Mabsadd L
.28952 .27755 1.875 0 Mabsadd L
.28952 .27755 0 -2.5 Mabsadd L
.28952 .27755 -1.875 0 Mabsadd L closepath F
.29048 .29239 -1.875 0 Mabsadd m
.29048 .29239 0 2.5 Mabsadd L
.29048 .29239 1.875 0 Mabsadd L
.29048 .29239 0 -2.5 Mabsadd L
.29048 .29239 -1.875 0 Mabsadd L closepath F
.29143 .2061 -1.875 0 Mabsadd m
.29143 .2061 0 2.5 Mabsadd L
.29143 .2061 1.875 0 Mabsadd L
.29143 .2061 0 -2.5 Mabsadd L
.29143 .2061 -1.875 0 Mabsadd L closepath F
.29238 .27953 -1.875 0 Mabsadd m
.29238 .27953 0 2.5 Mabsadd L
.29238 .27953 1.875 0 Mabsadd L
.29238 .27953 0 -2.5 Mabsadd L
.29238 .27953 -1.875 0 Mabsadd L closepath F
.29333 .30137 -1.875 0 Mabsadd m
.29333 .30137 0 2.5 Mabsadd L
.29333 .30137 1.875 0 Mabsadd L
.29333 .30137 0 -2.5 Mabsadd L
.29333 .30137 -1.875 0 Mabsadd L closepath F
.29429 .30966 -1.875 0 Mabsadd m
.29429 .30966 0 2.5 Mabsadd L
.29429 .30966 1.875 0 Mabsadd L
.29429 .30966 0 -2.5 Mabsadd L
.29429 .30966 -1.875 0 Mabsadd L closepath F
.29524 .31419 -1.875 0 Mabsadd m
.29524 .31419 0 2.5 Mabsadd L
.29524 .31419 1.875 0 Mabsadd L
.29524 .31419 0 -2.5 Mabsadd L
.29524 .31419 -1.875 0 Mabsadd L closepath F
.29619 .34396 -1.875 0 Mabsadd m
.29619 .34396 0 2.5 Mabsadd L
.29619 .34396 1.875 0 Mabsadd L
.29619 .34396 0 -2.5 Mabsadd L
.29619 .34396 -1.875 0 Mabsadd L closepath F
.29714 .34404 -1.875 0 Mabsadd m
.29714 .34404 0 2.5 Mabsadd L
.29714 .34404 1.875 0 Mabsadd L
.29714 .34404 0 -2.5 Mabsadd L
.29714 .34404 -1.875 0 Mabsadd L closepath F
.2981 .34429 -1.875 0 Mabsadd m
.2981 .34429 0 2.5 Mabsadd L
.2981 .34429 1.875 0 Mabsadd L
.2981 .34429 0 -2.5 Mabsadd L
.2981 .34429 -1.875 0 Mabsadd L closepath F
.29905 .34456 -1.875 0 Mabsadd m
.29905 .34456 0 2.5 Mabsadd L
.29905 .34456 1.875 0 Mabsadd L
.29905 .34456 0 -2.5 Mabsadd L
.29905 .34456 -1.875 0 Mabsadd L closepath F
.3 .34472 -1.875 0 Mabsadd m
.3 .34472 0 2.5 Mabsadd L
.3 .34472 1.875 0 Mabsadd L
.3 .34472 0 -2.5 Mabsadd L
.3 .34472 -1.875 0 Mabsadd L closepath F
.30095 .34479 -1.875 0 Mabsadd m
.30095 .34479 0 2.5 Mabsadd L
.30095 .34479 1.875 0 Mabsadd L
.30095 .34479 0 -2.5 Mabsadd L
.30095 .34479 -1.875 0 Mabsadd L closepath F
.3019 .34514 -1.875 0 Mabsadd m
.3019 .34514 0 2.5 Mabsadd L
.3019 .34514 1.875 0 Mabsadd L
.3019 .34514 0 -2.5 Mabsadd L
.3019 .34514 -1.875 0 Mabsadd L closepath F
.30286 .34605 -1.875 0 Mabsadd m
.30286 .34605 0 2.5 Mabsadd L
.30286 .34605 1.875 0 Mabsadd L
.30286 .34605 0 -2.5 Mabsadd L
.30286 .34605 -1.875 0 Mabsadd L closepath F
.30381 .3461 -1.875 0 Mabsadd m
.30381 .3461 0 2.5 Mabsadd L
.30381 .3461 1.875 0 Mabsadd L
.30381 .3461 0 -2.5 Mabsadd L
.30381 .3461 -1.875 0 Mabsadd L closepath F
.30476 .34624 -1.875 0 Mabsadd m
.30476 .34624 0 2.5 Mabsadd L
.30476 .34624 1.875 0 Mabsadd L
.30476 .34624 0 -2.5 Mabsadd L
.30476 .34624 -1.875 0 Mabsadd L closepath F
.30571 .34624 -1.875 0 Mabsadd m
.30571 .34624 0 2.5 Mabsadd L
.30571 .34624 1.875 0 Mabsadd L
.30571 .34624 0 -2.5 Mabsadd L
.30571 .34624 -1.875 0 Mabsadd L closepath F
.30667 .34627 -1.875 0 Mabsadd m
.30667 .34627 0 2.5 Mabsadd L
.30667 .34627 1.875 0 Mabsadd L
.30667 .34627 0 -2.5 Mabsadd L
.30667 .34627 -1.875 0 Mabsadd L closepath F
.30762 .34632 -1.875 0 Mabsadd m
.30762 .34632 0 2.5 Mabsadd L
.30762 .34632 1.875 0 Mabsadd L
.30762 .34632 0 -2.5 Mabsadd L
.30762 .34632 -1.875 0 Mabsadd L closepath F
.30857 .34632 -1.875 0 Mabsadd m
.30857 .34632 0 2.5 Mabsadd L
.30857 .34632 1.875 0 Mabsadd L
.30857 .34632 0 -2.5 Mabsadd L
.30857 .34632 -1.875 0 Mabsadd L closepath F
.30952 .34632 -1.875 0 Mabsadd m
.30952 .34632 0 2.5 Mabsadd L
.30952 .34632 1.875 0 Mabsadd L
.30952 .34632 0 -2.5 Mabsadd L
.30952 .34632 -1.875 0 Mabsadd L closepath F
.31048 .34633 -1.875 0 Mabsadd m
.31048 .34633 0 2.5 Mabsadd L
.31048 .34633 1.875 0 Mabsadd L
.31048 .34633 0 -2.5 Mabsadd L
.31048 .34633 -1.875 0 Mabsadd L closepath F
.31143 .34633 -1.875 0 Mabsadd m
.31143 .34633 0 2.5 Mabsadd L
.31143 .34633 1.875 0 Mabsadd L
.31143 .34633 0 -2.5 Mabsadd L
.31143 .34633 -1.875 0 Mabsadd L closepath F
.31238 .34634 -1.875 0 Mabsadd m
.31238 .34634 0 2.5 Mabsadd L
.31238 .34634 1.875 0 Mabsadd L
.31238 .34634 0 -2.5 Mabsadd L
.31238 .34634 -1.875 0 Mabsadd L closepath F
.31333 .34634 -1.875 0 Mabsadd m
.31333 .34634 0 2.5 Mabsadd L
.31333 .34634 1.875 0 Mabsadd L
.31333 .34634 0 -2.5 Mabsadd L
.31333 .34634 -1.875 0 Mabsadd L closepath F
.31429 .34634 -1.875 0 Mabsadd m
.31429 .34634 0 2.5 Mabsadd L
.31429 .34634 1.875 0 Mabsadd L
.31429 .34634 0 -2.5 Mabsadd L
.31429 .34634 -1.875 0 Mabsadd L closepath F
.31524 .34634 -1.875 0 Mabsadd m
.31524 .34634 0 2.5 Mabsadd L
.31524 .34634 1.875 0 Mabsadd L
.31524 .34634 0 -2.5 Mabsadd L
.31524 .34634 -1.875 0 Mabsadd L closepath F
.31619 .34634 -1.875 0 Mabsadd m
.31619 .34634 0 2.5 Mabsadd L
.31619 .34634 1.875 0 Mabsadd L
.31619 .34634 0 -2.5 Mabsadd L
.31619 .34634 -1.875 0 Mabsadd L closepath F
.31714 .34634 -1.875 0 Mabsadd m
.31714 .34634 0 2.5 Mabsadd L
.31714 .34634 1.875 0 Mabsadd L
.31714 .34634 0 -2.5 Mabsadd L
.31714 .34634 -1.875 0 Mabsadd L closepath F
.3181 .34634 -1.875 0 Mabsadd m
.3181 .34634 0 2.5 Mabsadd L
.3181 .34634 1.875 0 Mabsadd L
.3181 .34634 0 -2.5 Mabsadd L
.3181 .34634 -1.875 0 Mabsadd L closepath F
.31905 .34634 -1.875 0 Mabsadd m
.31905 .34634 0 2.5 Mabsadd L
.31905 .34634 1.875 0 Mabsadd L
.31905 .34634 0 -2.5 Mabsadd L
.31905 .34634 -1.875 0 Mabsadd L closepath F
.32 .34634 -1.875 0 Mabsadd m
.32 .34634 0 2.5 Mabsadd L
.32 .34634 1.875 0 Mabsadd L
.32 .34634 0 -2.5 Mabsadd L
.32 .34634 -1.875 0 Mabsadd L closepath F
.32095 .34634 -1.875 0 Mabsadd m
.32095 .34634 0 2.5 Mabsadd L
.32095 .34634 1.875 0 Mabsadd L
.32095 .34634 0 -2.5 Mabsadd L
.32095 .34634 -1.875 0 Mabsadd L closepath F
.3219 .34634 -1.875 0 Mabsadd m
.3219 .34634 0 2.5 Mabsadd L
.3219 .34634 1.875 0 Mabsadd L
.3219 .34634 0 -2.5 Mabsadd L
.3219 .34634 -1.875 0 Mabsadd L closepath F
.32286 .34634 -1.875 0 Mabsadd m
.32286 .34634 0 2.5 Mabsadd L
.32286 .34634 1.875 0 Mabsadd L
.32286 .34634 0 -2.5 Mabsadd L
.32286 .34634 -1.875 0 Mabsadd L closepath F
.32381 .34634 -1.875 0 Mabsadd m
.32381 .34634 0 2.5 Mabsadd L
.32381 .34634 1.875 0 Mabsadd L
.32381 .34634 0 -2.5 Mabsadd L
.32381 .34634 -1.875 0 Mabsadd L closepath F
.32476 .34633 -1.875 0 Mabsadd m
.32476 .34633 0 2.5 Mabsadd L
.32476 .34633 1.875 0 Mabsadd L
.32476 .34633 0 -2.5 Mabsadd L
.32476 .34633 -1.875 0 Mabsadd L closepath F
.32571 .34633 -1.875 0 Mabsadd m
.32571 .34633 0 2.5 Mabsadd L
.32571 .34633 1.875 0 Mabsadd L
.32571 .34633 0 -2.5 Mabsadd L
.32571 .34633 -1.875 0 Mabsadd L closepath F
.32667 .34633 -1.875 0 Mabsadd m
.32667 .34633 0 2.5 Mabsadd L
.32667 .34633 1.875 0 Mabsadd L
.32667 .34633 0 -2.5 Mabsadd L
.32667 .34633 -1.875 0 Mabsadd L closepath F
.32762 .34633 -1.875 0 Mabsadd m
.32762 .34633 0 2.5 Mabsadd L
.32762 .34633 1.875 0 Mabsadd L
.32762 .34633 0 -2.5 Mabsadd L
.32762 .34633 -1.875 0 Mabsadd L closepath F
.32857 .34632 -1.875 0 Mabsadd m
.32857 .34632 0 2.5 Mabsadd L
.32857 .34632 1.875 0 Mabsadd L
.32857 .34632 0 -2.5 Mabsadd L
.32857 .34632 -1.875 0 Mabsadd L closepath F
.32952 .34632 -1.875 0 Mabsadd m
.32952 .34632 0 2.5 Mabsadd L
.32952 .34632 1.875 0 Mabsadd L
.32952 .34632 0 -2.5 Mabsadd L
.32952 .34632 -1.875 0 Mabsadd L closepath F
.33048 .34629 -1.875 0 Mabsadd m
.33048 .34629 0 2.5 Mabsadd L
.33048 .34629 1.875 0 Mabsadd L
.33048 .34629 0 -2.5 Mabsadd L
.33048 .34629 -1.875 0 Mabsadd L closepath F
.33143 .34629 -1.875 0 Mabsadd m
.33143 .34629 0 2.5 Mabsadd L
.33143 .34629 1.875 0 Mabsadd L
.33143 .34629 0 -2.5 Mabsadd L
.33143 .34629 -1.875 0 Mabsadd L closepath F
.33238 .34624 -1.875 0 Mabsadd m
.33238 .34624 0 2.5 Mabsadd L
.33238 .34624 1.875 0 Mabsadd L
.33238 .34624 0 -2.5 Mabsadd L
.33238 .34624 -1.875 0 Mabsadd L closepath F
.33333 .34622 -1.875 0 Mabsadd m
.33333 .34622 0 2.5 Mabsadd L
.33333 .34622 1.875 0 Mabsadd L
.33333 .34622 0 -2.5 Mabsadd L
.33333 .34622 -1.875 0 Mabsadd L closepath F
.33429 .34594 -1.875 0 Mabsadd m
.33429 .34594 0 2.5 Mabsadd L
.33429 .34594 1.875 0 Mabsadd L
.33429 .34594 0 -2.5 Mabsadd L
.33429 .34594 -1.875 0 Mabsadd L closepath F
.33524 .34577 -1.875 0 Mabsadd m
.33524 .34577 0 2.5 Mabsadd L
.33524 .34577 1.875 0 Mabsadd L
.33524 .34577 0 -2.5 Mabsadd L
.33524 .34577 -1.875 0 Mabsadd L closepath F
.33619 .3433 -1.875 0 Mabsadd m
.33619 .3433 0 2.5 Mabsadd L
.33619 .3433 1.875 0 Mabsadd L
.33619 .3433 0 -2.5 Mabsadd L
.33619 .3433 -1.875 0 Mabsadd L closepath F
.33714 .34106 -1.875 0 Mabsadd m
.33714 .34106 0 2.5 Mabsadd L
.33714 .34106 1.875 0 Mabsadd L
.33714 .34106 0 -2.5 Mabsadd L
.33714 .34106 -1.875 0 Mabsadd L closepath F
.3381 .33091 -1.875 0 Mabsadd m
.3381 .33091 0 2.5 Mabsadd L
.3381 .33091 1.875 0 Mabsadd L
.3381 .33091 0 -2.5 Mabsadd L
.3381 .33091 -1.875 0 Mabsadd L closepath F
.33905 .27786 -1.875 0 Mabsadd m
.33905 .27786 0 2.5 Mabsadd L
.33905 .27786 1.875 0 Mabsadd L
.33905 .27786 0 -2.5 Mabsadd L
.33905 .27786 -1.875 0 Mabsadd L closepath F
.34 .27786 -1.875 0 Mabsadd m
.34 .27786 0 2.5 Mabsadd L
.34 .27786 1.875 0 Mabsadd L
.34 .27786 0 -2.5 Mabsadd L
.34 .27786 -1.875 0 Mabsadd L closepath F
.34095 .28049 -1.875 0 Mabsadd m
.34095 .28049 0 2.5 Mabsadd L
.34095 .28049 1.875 0 Mabsadd L
.34095 .28049 0 -2.5 Mabsadd L
.34095 .28049 -1.875 0 Mabsadd L closepath F
.3419 .28348 -1.875 0 Mabsadd m
.3419 .28348 0 2.5 Mabsadd L
.3419 .28348 1.875 0 Mabsadd L
.3419 .28348 0 -2.5 Mabsadd L
.3419 .28348 -1.875 0 Mabsadd L closepath F
.34286 .25595 -1.875 0 Mabsadd m
.34286 .25595 0 2.5 Mabsadd L
.34286 .25595 1.875 0 Mabsadd L
.34286 .25595 0 -2.5 Mabsadd L
.34286 .25595 -1.875 0 Mabsadd L closepath F
.34381 .27742 -1.875 0 Mabsadd m
.34381 .27742 0 2.5 Mabsadd L
.34381 .27742 1.875 0 Mabsadd L
.34381 .27742 0 -2.5 Mabsadd L
.34381 .27742 -1.875 0 Mabsadd L closepath F
.34476 .12474 -1.875 0 Mabsadd m
.34476 .12474 0 2.5 Mabsadd L
.34476 .12474 1.875 0 Mabsadd L
.34476 .12474 0 -2.5 Mabsadd L
.34476 .12474 -1.875 0 Mabsadd L closepath F
.34571 .25416 -1.875 0 Mabsadd m
.34571 .25416 0 2.5 Mabsadd L
.34571 .25416 1.875 0 Mabsadd L
.34571 .25416 0 -2.5 Mabsadd L
.34571 .25416 -1.875 0 Mabsadd L closepath F
.34667 .26684 -1.875 0 Mabsadd m
.34667 .26684 0 2.5 Mabsadd L
.34667 .26684 1.875 0 Mabsadd L
.34667 .26684 0 -2.5 Mabsadd L
.34667 .26684 -1.875 0 Mabsadd L closepath F
.34762 .25617 -1.875 0 Mabsadd m
.34762 .25617 0 2.5 Mabsadd L
.34762 .25617 1.875 0 Mabsadd L
.34762 .25617 0 -2.5 Mabsadd L
.34762 .25617 -1.875 0 Mabsadd L closepath F
.34857 .26226 -1.875 0 Mabsadd m
.34857 .26226 0 2.5 Mabsadd L
.34857 .26226 1.875 0 Mabsadd L
.34857 .26226 0 -2.5 Mabsadd L
.34857 .26226 -1.875 0 Mabsadd L closepath F
.34952 .26286 -1.875 0 Mabsadd m
.34952 .26286 0 2.5 Mabsadd L
.34952 .26286 1.875 0 Mabsadd L
.34952 .26286 0 -2.5 Mabsadd L
.34952 .26286 -1.875 0 Mabsadd L closepath F
.35048 .266 -1.875 0 Mabsadd m
.35048 .266 0 2.5 Mabsadd L
.35048 .266 1.875 0 Mabsadd L
.35048 .266 0 -2.5 Mabsadd L
.35048 .266 -1.875 0 Mabsadd L closepath F
.35143 .26887 -1.875 0 Mabsadd m
.35143 .26887 0 2.5 Mabsadd L
.35143 .26887 1.875 0 Mabsadd L
.35143 .26887 0 -2.5 Mabsadd L
.35143 .26887 -1.875 0 Mabsadd L closepath F
.35238 .27027 -1.875 0 Mabsadd m
.35238 .27027 0 2.5 Mabsadd L
.35238 .27027 1.875 0 Mabsadd L
.35238 .27027 0 -2.5 Mabsadd L
.35238 .27027 -1.875 0 Mabsadd L closepath F
.35333 .27565 -1.875 0 Mabsadd m
.35333 .27565 0 2.5 Mabsadd L
.35333 .27565 1.875 0 Mabsadd L
.35333 .27565 0 -2.5 Mabsadd L
.35333 .27565 -1.875 0 Mabsadd L closepath F
.35429 .27708 -1.875 0 Mabsadd m
.35429 .27708 0 2.5 Mabsadd L
.35429 .27708 1.875 0 Mabsadd L
.35429 .27708 0 -2.5 Mabsadd L
.35429 .27708 -1.875 0 Mabsadd L closepath F
.35524 .28035 -1.875 0 Mabsadd m
.35524 .28035 0 2.5 Mabsadd L
.35524 .28035 1.875 0 Mabsadd L
.35524 .28035 0 -2.5 Mabsadd L
.35524 .28035 -1.875 0 Mabsadd L closepath F
.35619 .28049 -1.875 0 Mabsadd m
.35619 .28049 0 2.5 Mabsadd L
.35619 .28049 1.875 0 Mabsadd L
.35619 .28049 0 -2.5 Mabsadd L
.35619 .28049 -1.875 0 Mabsadd L closepath F
.35714 .28087 -1.875 0 Mabsadd m
.35714 .28087 0 2.5 Mabsadd L
.35714 .28087 1.875 0 Mabsadd L
.35714 .28087 0 -2.5 Mabsadd L
.35714 .28087 -1.875 0 Mabsadd L closepath F
.3581 .28089 -1.875 0 Mabsadd m
.3581 .28089 0 2.5 Mabsadd L
.3581 .28089 1.875 0 Mabsadd L
.3581 .28089 0 -2.5 Mabsadd L
.3581 .28089 -1.875 0 Mabsadd L closepath F
.35905 .28096 -1.875 0 Mabsadd m
.35905 .28096 0 2.5 Mabsadd L
.35905 .28096 1.875 0 Mabsadd L
.35905 .28096 0 -2.5 Mabsadd L
.35905 .28096 -1.875 0 Mabsadd L closepath F
.36 .28106 -1.875 0 Mabsadd m
.36 .28106 0 2.5 Mabsadd L
.36 .28106 1.875 0 Mabsadd L
.36 .28106 0 -2.5 Mabsadd L
.36 .28106 -1.875 0 Mabsadd L closepath F
.36095 .28106 -1.875 0 Mabsadd m
.36095 .28106 0 2.5 Mabsadd L
.36095 .28106 1.875 0 Mabsadd L
.36095 .28106 0 -2.5 Mabsadd L
.36095 .28106 -1.875 0 Mabsadd L closepath F
.3619 .28107 -1.875 0 Mabsadd m
.3619 .28107 0 2.5 Mabsadd L
.3619 .28107 1.875 0 Mabsadd L
.3619 .28107 0 -2.5 Mabsadd L
.3619 .28107 -1.875 0 Mabsadd L closepath F
.36286 .28108 -1.875 0 Mabsadd m
.36286 .28108 0 2.5 Mabsadd L
.36286 .28108 1.875 0 Mabsadd L
.36286 .28108 0 -2.5 Mabsadd L
.36286 .28108 -1.875 0 Mabsadd L closepath F
.36381 .28108 -1.875 0 Mabsadd m
.36381 .28108 0 2.5 Mabsadd L
.36381 .28108 1.875 0 Mabsadd L
.36381 .28108 0 -2.5 Mabsadd L
.36381 .28108 -1.875 0 Mabsadd L closepath F
.36476 .28109 -1.875 0 Mabsadd m
.36476 .28109 0 2.5 Mabsadd L
.36476 .28109 1.875 0 Mabsadd L
.36476 .28109 0 -2.5 Mabsadd L
.36476 .28109 -1.875 0 Mabsadd L closepath F
.36571 .28109 -1.875 0 Mabsadd m
.36571 .28109 0 2.5 Mabsadd L
.36571 .28109 1.875 0 Mabsadd L
.36571 .28109 0 -2.5 Mabsadd L
.36571 .28109 -1.875 0 Mabsadd L closepath F
.36667 .28109 -1.875 0 Mabsadd m
.36667 .28109 0 2.5 Mabsadd L
.36667 .28109 1.875 0 Mabsadd L
.36667 .28109 0 -2.5 Mabsadd L
.36667 .28109 -1.875 0 Mabsadd L closepath F
.36762 .28109 -1.875 0 Mabsadd m
.36762 .28109 0 2.5 Mabsadd L
.36762 .28109 1.875 0 Mabsadd L
.36762 .28109 0 -2.5 Mabsadd L
.36762 .28109 -1.875 0 Mabsadd L closepath F
.36857 .28109 -1.875 0 Mabsadd m
.36857 .28109 0 2.5 Mabsadd L
.36857 .28109 1.875 0 Mabsadd L
.36857 .28109 0 -2.5 Mabsadd L
.36857 .28109 -1.875 0 Mabsadd L closepath F
.36952 .28109 -1.875 0 Mabsadd m
.36952 .28109 0 2.5 Mabsadd L
.36952 .28109 1.875 0 Mabsadd L
.36952 .28109 0 -2.5 Mabsadd L
.36952 .28109 -1.875 0 Mabsadd L closepath F
.37048 .28109 -1.875 0 Mabsadd m
.37048 .28109 0 2.5 Mabsadd L
.37048 .28109 1.875 0 Mabsadd L
.37048 .28109 0 -2.5 Mabsadd L
.37048 .28109 -1.875 0 Mabsadd L closepath F
.37143 .28109 -1.875 0 Mabsadd m
.37143 .28109 0 2.5 Mabsadd L
.37143 .28109 1.875 0 Mabsadd L
.37143 .28109 0 -2.5 Mabsadd L
.37143 .28109 -1.875 0 Mabsadd L closepath F
.37238 .28109 -1.875 0 Mabsadd m
.37238 .28109 0 2.5 Mabsadd L
.37238 .28109 1.875 0 Mabsadd L
.37238 .28109 0 -2.5 Mabsadd L
.37238 .28109 -1.875 0 Mabsadd L closepath F
.37333 .28109 -1.875 0 Mabsadd m
.37333 .28109 0 2.5 Mabsadd L
.37333 .28109 1.875 0 Mabsadd L
.37333 .28109 0 -2.5 Mabsadd L
.37333 .28109 -1.875 0 Mabsadd L closepath F
.37429 .28109 -1.875 0 Mabsadd m
.37429 .28109 0 2.5 Mabsadd L
.37429 .28109 1.875 0 Mabsadd L
.37429 .28109 0 -2.5 Mabsadd L
.37429 .28109 -1.875 0 Mabsadd L closepath F
.37524 .28109 -1.875 0 Mabsadd m
.37524 .28109 0 2.5 Mabsadd L
.37524 .28109 1.875 0 Mabsadd L
.37524 .28109 0 -2.5 Mabsadd L
.37524 .28109 -1.875 0 Mabsadd L closepath F
.37619 .28109 -1.875 0 Mabsadd m
.37619 .28109 0 2.5 Mabsadd L
.37619 .28109 1.875 0 Mabsadd L
.37619 .28109 0 -2.5 Mabsadd L
.37619 .28109 -1.875 0 Mabsadd L closepath F
.37714 .28109 -1.875 0 Mabsadd m
.37714 .28109 0 2.5 Mabsadd L
.37714 .28109 1.875 0 Mabsadd L
.37714 .28109 0 -2.5 Mabsadd L
.37714 .28109 -1.875 0 Mabsadd L closepath F
.3781 .28109 -1.875 0 Mabsadd m
.3781 .28109 0 2.5 Mabsadd L
.3781 .28109 1.875 0 Mabsadd L
.3781 .28109 0 -2.5 Mabsadd L
.3781 .28109 -1.875 0 Mabsadd L closepath F
.37905 .28108 -1.875 0 Mabsadd m
.37905 .28108 0 2.5 Mabsadd L
.37905 .28108 1.875 0 Mabsadd L
.37905 .28108 0 -2.5 Mabsadd L
.37905 .28108 -1.875 0 Mabsadd L closepath F
.38 .28108 -1.875 0 Mabsadd m
.38 .28108 0 2.5 Mabsadd L
.38 .28108 1.875 0 Mabsadd L
.38 .28108 0 -2.5 Mabsadd L
.38 .28108 -1.875 0 Mabsadd L closepath F
.38095 .28108 -1.875 0 Mabsadd m
.38095 .28108 0 2.5 Mabsadd L
.38095 .28108 1.875 0 Mabsadd L
.38095 .28108 0 -2.5 Mabsadd L
.38095 .28108 -1.875 0 Mabsadd L closepath F
.3819 .28107 -1.875 0 Mabsadd m
.3819 .28107 0 2.5 Mabsadd L
.3819 .28107 1.875 0 Mabsadd L
.3819 .28107 0 -2.5 Mabsadd L
.3819 .28107 -1.875 0 Mabsadd L closepath F
.38286 .28107 -1.875 0 Mabsadd m
.38286 .28107 0 2.5 Mabsadd L
.38286 .28107 1.875 0 Mabsadd L
.38286 .28107 0 -2.5 Mabsadd L
.38286 .28107 -1.875 0 Mabsadd L closepath F
.38381 .28106 -1.875 0 Mabsadd m
.38381 .28106 0 2.5 Mabsadd L
.38381 .28106 1.875 0 Mabsadd L
.38381 .28106 0 -2.5 Mabsadd L
.38381 .28106 -1.875 0 Mabsadd L closepath F
.38476 .28102 -1.875 0 Mabsadd m
.38476 .28102 0 2.5 Mabsadd L
.38476 .28102 1.875 0 Mabsadd L
.38476 .28102 0 -2.5 Mabsadd L
.38476 .28102 -1.875 0 Mabsadd L closepath F
.38571 .28101 -1.875 0 Mabsadd m
.38571 .28101 0 2.5 Mabsadd L
.38571 .28101 1.875 0 Mabsadd L
.38571 .28101 0 -2.5 Mabsadd L
.38571 .28101 -1.875 0 Mabsadd L closepath F
.38667 .28091 -1.875 0 Mabsadd m
.38667 .28091 0 2.5 Mabsadd L
.38667 .28091 1.875 0 Mabsadd L
.38667 .28091 0 -2.5 Mabsadd L
.38667 .28091 -1.875 0 Mabsadd L closepath F
.38762 .28086 -1.875 0 Mabsadd m
.38762 .28086 0 2.5 Mabsadd L
.38762 .28086 1.875 0 Mabsadd L
.38762 .28086 0 -2.5 Mabsadd L
.38762 .28086 -1.875 0 Mabsadd L closepath F
.38857 .28019 -1.875 0 Mabsadd m
.38857 .28019 0 2.5 Mabsadd L
.38857 .28019 1.875 0 Mabsadd L
.38857 .28019 0 -2.5 Mabsadd L
.38857 .28019 -1.875 0 Mabsadd L closepath F
.38952 .27969 -1.875 0 Mabsadd m
.38952 .27969 0 2.5 Mabsadd L
.38952 .27969 1.875 0 Mabsadd L
.38952 .27969 0 -2.5 Mabsadd L
.38952 .27969 -1.875 0 Mabsadd L closepath F
.39048 .2778 -1.875 0 Mabsadd m
.39048 .2778 0 2.5 Mabsadd L
.39048 .2778 1.875 0 Mabsadd L
.39048 .2778 0 -2.5 Mabsadd L
.39048 .2778 -1.875 0 Mabsadd L closepath F
.39143 .26951 -1.875 0 Mabsadd m
.39143 .26951 0 2.5 Mabsadd L
.39143 .26951 1.875 0 Mabsadd L
.39143 .26951 0 -2.5 Mabsadd L
.39143 .26951 -1.875 0 Mabsadd L closepath F
.39238 .22707 -1.875 0 Mabsadd m
.39238 .22707 0 2.5 Mabsadd L
.39238 .22707 1.875 0 Mabsadd L
.39238 .22707 0 -2.5 Mabsadd L
.39238 .22707 -1.875 0 Mabsadd L closepath F
.39333 .22706 -1.875 0 Mabsadd m
.39333 .22706 0 2.5 Mabsadd L
.39333 .22706 1.875 0 Mabsadd L
.39333 .22706 0 -2.5 Mabsadd L
.39333 .22706 -1.875 0 Mabsadd L closepath F
.39429 .22907 -1.875 0 Mabsadd m
.39429 .22907 0 2.5 Mabsadd L
.39429 .22907 1.875 0 Mabsadd L
.39429 .22907 0 -2.5 Mabsadd L
.39429 .22907 -1.875 0 Mabsadd L closepath F
.39524 .23133 -1.875 0 Mabsadd m
.39524 .23133 0 2.5 Mabsadd L
.39524 .23133 1.875 0 Mabsadd L
.39524 .23133 0 -2.5 Mabsadd L
.39524 .23133 -1.875 0 Mabsadd L closepath F
.39619 .20748 -1.875 0 Mabsadd m
.39619 .20748 0 2.5 Mabsadd L
.39619 .20748 1.875 0 Mabsadd L
.39619 .20748 0 -2.5 Mabsadd L
.39619 .20748 -1.875 0 Mabsadd L closepath F
.39714 .22355 -1.875 0 Mabsadd m
.39714 .22355 0 2.5 Mabsadd L
.39714 .22355 1.875 0 Mabsadd L
.39714 .22355 0 -2.5 Mabsadd L
.39714 .22355 -1.875 0 Mabsadd L closepath F
.3981 .04194 -1.875 0 Mabsadd m
.3981 .04194 0 2.5 Mabsadd L
.3981 .04194 1.875 0 Mabsadd L
.3981 .04194 0 -2.5 Mabsadd L
.3981 .04194 -1.875 0 Mabsadd L closepath F
.39905 .22193 -1.875 0 Mabsadd m
.39905 .22193 0 2.5 Mabsadd L
.39905 .22193 1.875 0 Mabsadd L
.39905 .22193 0 -2.5 Mabsadd L
.39905 .22193 -1.875 0 Mabsadd L closepath F
.4 .22256 -1.875 0 Mabsadd m
.4 .22256 0 2.5 Mabsadd L
.4 .22256 1.875 0 Mabsadd L
.4 .22256 0 -2.5 Mabsadd L
.4 .22256 -1.875 0 Mabsadd L closepath F
.40095 .17182 -1.875 0 Mabsadd m
.40095 .17182 0 2.5 Mabsadd L
.40095 .17182 1.875 0 Mabsadd L
.40095 .17182 0 -2.5 Mabsadd L
.40095 .17182 -1.875 0 Mabsadd L closepath F
.4019 .1799 -1.875 0 Mabsadd m
.4019 .1799 0 2.5 Mabsadd L
.4019 .1799 1.875 0 Mabsadd L
.4019 .1799 0 -2.5 Mabsadd L
.4019 .1799 -1.875 0 Mabsadd L closepath F
.40286 .18071 -1.875 0 Mabsadd m
.40286 .18071 0 2.5 Mabsadd L
.40286 .18071 1.875 0 Mabsadd L
.40286 .18071 0 -2.5 Mabsadd L
.40286 .18071 -1.875 0 Mabsadd L closepath F
.40381 .18501 -1.875 0 Mabsadd m
.40381 .18501 0 2.5 Mabsadd L
.40381 .18501 1.875 0 Mabsadd L
.40381 .18501 0 -2.5 Mabsadd L
.40381 .18501 -1.875 0 Mabsadd L closepath F
.40476 .18888 -1.875 0 Mabsadd m
.40476 .18888 0 2.5 Mabsadd L
.40476 .18888 1.875 0 Mabsadd L
.40476 .18888 0 -2.5 Mabsadd L
.40476 .18888 -1.875 0 Mabsadd L closepath F
.40571 .21808 -1.875 0 Mabsadd m
.40571 .21808 0 2.5 Mabsadd L
.40571 .21808 1.875 0 Mabsadd L
.40571 .21808 0 -2.5 Mabsadd L
.40571 .21808 -1.875 0 Mabsadd L closepath F
.40667 .22486 -1.875 0 Mabsadd m
.40667 .22486 0 2.5 Mabsadd L
.40667 .22486 1.875 0 Mabsadd L
.40667 .22486 0 -2.5 Mabsadd L
.40667 .22486 -1.875 0 Mabsadd L closepath F
.40762 .22656 -1.875 0 Mabsadd m
.40762 .22656 0 2.5 Mabsadd L
.40762 .22656 1.875 0 Mabsadd L
.40762 .22656 0 -2.5 Mabsadd L
.40762 .22656 -1.875 0 Mabsadd L closepath F
.40857 .23038 -1.875 0 Mabsadd m
.40857 .23038 0 2.5 Mabsadd L
.40857 .23038 1.875 0 Mabsadd L
.40857 .23038 0 -2.5 Mabsadd L
.40857 .23038 -1.875 0 Mabsadd L closepath F
.40952 .23054 -1.875 0 Mabsadd m
.40952 .23054 0 2.5 Mabsadd L
.40952 .23054 1.875 0 Mabsadd L
.40952 .23054 0 -2.5 Mabsadd L
.40952 .23054 -1.875 0 Mabsadd L closepath F
.41048 .23097 -1.875 0 Mabsadd m
.41048 .23097 0 2.5 Mabsadd L
.41048 .23097 1.875 0 Mabsadd L
.41048 .23097 0 -2.5 Mabsadd L
.41048 .23097 -1.875 0 Mabsadd L closepath F
.41143 .23099 -1.875 0 Mabsadd m
.41143 .23099 0 2.5 Mabsadd L
.41143 .23099 1.875 0 Mabsadd L
.41143 .23099 0 -2.5 Mabsadd L
.41143 .23099 -1.875 0 Mabsadd L closepath F
.41238 .23107 -1.875 0 Mabsadd m
.41238 .23107 0 2.5 Mabsadd L
.41238 .23107 1.875 0 Mabsadd L
.41238 .23107 0 -2.5 Mabsadd L
.41238 .23107 -1.875 0 Mabsadd L closepath F
.41333 .23118 -1.875 0 Mabsadd m
.41333 .23118 0 2.5 Mabsadd L
.41333 .23118 1.875 0 Mabsadd L
.41333 .23118 0 -2.5 Mabsadd L
.41333 .23118 -1.875 0 Mabsadd L closepath F
.41429 .23118 -1.875 0 Mabsadd m
.41429 .23118 0 2.5 Mabsadd L
.41429 .23118 1.875 0 Mabsadd L
.41429 .23118 0 -2.5 Mabsadd L
.41429 .23118 -1.875 0 Mabsadd L closepath F
.41524 .23119 -1.875 0 Mabsadd m
.41524 .23119 0 2.5 Mabsadd L
.41524 .23119 1.875 0 Mabsadd L
.41524 .23119 0 -2.5 Mabsadd L
.41524 .23119 -1.875 0 Mabsadd L closepath F
.41619 .23121 -1.875 0 Mabsadd m
.41619 .23121 0 2.5 Mabsadd L
.41619 .23121 1.875 0 Mabsadd L
.41619 .23121 0 -2.5 Mabsadd L
.41619 .23121 -1.875 0 Mabsadd L closepath F
.41714 .23121 -1.875 0 Mabsadd m
.41714 .23121 0 2.5 Mabsadd L
.41714 .23121 1.875 0 Mabsadd L
.41714 .23121 0 -2.5 Mabsadd L
.41714 .23121 -1.875 0 Mabsadd L closepath F
.4181 .23121 -1.875 0 Mabsadd m
.4181 .23121 0 2.5 Mabsadd L
.4181 .23121 1.875 0 Mabsadd L
.4181 .23121 0 -2.5 Mabsadd L
.4181 .23121 -1.875 0 Mabsadd L closepath F
.41905 .23121 -1.875 0 Mabsadd m
.41905 .23121 0 2.5 Mabsadd L
.41905 .23121 1.875 0 Mabsadd L
.41905 .23121 0 -2.5 Mabsadd L
.41905 .23121 -1.875 0 Mabsadd L closepath F
.42 .23121 -1.875 0 Mabsadd m
.42 .23121 0 2.5 Mabsadd L
.42 .23121 1.875 0 Mabsadd L
.42 .23121 0 -2.5 Mabsadd L
.42 .23121 -1.875 0 Mabsadd L closepath F
.42095 .23121 -1.875 0 Mabsadd m
.42095 .23121 0 2.5 Mabsadd L
.42095 .23121 1.875 0 Mabsadd L
.42095 .23121 0 -2.5 Mabsadd L
.42095 .23121 -1.875 0 Mabsadd L closepath F
.4219 .23121 -1.875 0 Mabsadd m
.4219 .23121 0 2.5 Mabsadd L
.4219 .23121 1.875 0 Mabsadd L
.4219 .23121 0 -2.5 Mabsadd L
.4219 .23121 -1.875 0 Mabsadd L closepath F
.42286 .23122 -1.875 0 Mabsadd m
.42286 .23122 0 2.5 Mabsadd L
.42286 .23122 1.875 0 Mabsadd L
.42286 .23122 0 -2.5 Mabsadd L
.42286 .23122 -1.875 0 Mabsadd L closepath F
.42381 .23122 -1.875 0 Mabsadd m
.42381 .23122 0 2.5 Mabsadd L
.42381 .23122 1.875 0 Mabsadd L
.42381 .23122 0 -2.5 Mabsadd L
.42381 .23122 -1.875 0 Mabsadd L closepath F
.42476 .23122 -1.875 0 Mabsadd m
.42476 .23122 0 2.5 Mabsadd L
.42476 .23122 1.875 0 Mabsadd L
.42476 .23122 0 -2.5 Mabsadd L
.42476 .23122 -1.875 0 Mabsadd L closepath F
.42571 .23122 -1.875 0 Mabsadd m
.42571 .23122 0 2.5 Mabsadd L
.42571 .23122 1.875 0 Mabsadd L
.42571 .23122 0 -2.5 Mabsadd L
.42571 .23122 -1.875 0 Mabsadd L closepath F
.42667 .23122 -1.875 0 Mabsadd m
.42667 .23122 0 2.5 Mabsadd L
.42667 .23122 1.875 0 Mabsadd L
.42667 .23122 0 -2.5 Mabsadd L
.42667 .23122 -1.875 0 Mabsadd L closepath F
.42762 .23122 -1.875 0 Mabsadd m
.42762 .23122 0 2.5 Mabsadd L
.42762 .23122 1.875 0 Mabsadd L
.42762 .23122 0 -2.5 Mabsadd L
.42762 .23122 -1.875 0 Mabsadd L closepath F
.42857 .23121 -1.875 0 Mabsadd m
.42857 .23121 0 2.5 Mabsadd L
.42857 .23121 1.875 0 Mabsadd L
.42857 .23121 0 -2.5 Mabsadd L
.42857 .23121 -1.875 0 Mabsadd L closepath F
.42952 .23121 -1.875 0 Mabsadd m
.42952 .23121 0 2.5 Mabsadd L
.42952 .23121 1.875 0 Mabsadd L
.42952 .23121 0 -2.5 Mabsadd L
.42952 .23121 -1.875 0 Mabsadd L closepath F
.43048 .23121 -1.875 0 Mabsadd m
.43048 .23121 0 2.5 Mabsadd L
.43048 .23121 1.875 0 Mabsadd L
.43048 .23121 0 -2.5 Mabsadd L
.43048 .23121 -1.875 0 Mabsadd L closepath F
.43143 .23121 -1.875 0 Mabsadd m
.43143 .23121 0 2.5 Mabsadd L
.43143 .23121 1.875 0 Mabsadd L
.43143 .23121 0 -2.5 Mabsadd L
.43143 .23121 -1.875 0 Mabsadd L closepath F
.43238 .23121 -1.875 0 Mabsadd m
.43238 .23121 0 2.5 Mabsadd L
.43238 .23121 1.875 0 Mabsadd L
.43238 .23121 0 -2.5 Mabsadd L
.43238 .23121 -1.875 0 Mabsadd L closepath F
.43333 .23121 -1.875 0 Mabsadd m
.43333 .23121 0 2.5 Mabsadd L
.43333 .23121 1.875 0 Mabsadd L
.43333 .23121 0 -2.5 Mabsadd L
.43333 .23121 -1.875 0 Mabsadd L closepath F
.43429 .23121 -1.875 0 Mabsadd m
.43429 .23121 0 2.5 Mabsadd L
.43429 .23121 1.875 0 Mabsadd L
.43429 .23121 0 -2.5 Mabsadd L
.43429 .23121 -1.875 0 Mabsadd L closepath F
.43524 .2312 -1.875 0 Mabsadd m
.43524 .2312 0 2.5 Mabsadd L
.43524 .2312 1.875 0 Mabsadd L
.43524 .2312 0 -2.5 Mabsadd L
.43524 .2312 -1.875 0 Mabsadd L closepath F
.43619 .2312 -1.875 0 Mabsadd m
.43619 .2312 0 2.5 Mabsadd L
.43619 .2312 1.875 0 Mabsadd L
.43619 .2312 0 -2.5 Mabsadd L
.43619 .2312 -1.875 0 Mabsadd L closepath F
.43714 .23119 -1.875 0 Mabsadd m
.43714 .23119 0 2.5 Mabsadd L
.43714 .23119 1.875 0 Mabsadd L
.43714 .23119 0 -2.5 Mabsadd L
.43714 .23119 -1.875 0 Mabsadd L closepath F
.4381 .23115 -1.875 0 Mabsadd m
.4381 .23115 0 2.5 Mabsadd L
.4381 .23115 1.875 0 Mabsadd L
.4381 .23115 0 -2.5 Mabsadd L
.4381 .23115 -1.875 0 Mabsadd L closepath F
.43905 .23114 -1.875 0 Mabsadd m
.43905 .23114 0 2.5 Mabsadd L
.43905 .23114 1.875 0 Mabsadd L
.43905 .23114 0 -2.5 Mabsadd L
.43905 .23114 -1.875 0 Mabsadd L closepath F
.44 .23105 -1.875 0 Mabsadd m
.44 .23105 0 2.5 Mabsadd L
.44 .23105 1.875 0 Mabsadd L
.44 .23105 0 -2.5 Mabsadd L
.44 .23105 -1.875 0 Mabsadd L closepath F
.44095 .231 -1.875 0 Mabsadd m
.44095 .231 0 2.5 Mabsadd L
.44095 .231 1.875 0 Mabsadd L
.44095 .231 0 -2.5 Mabsadd L
.44095 .231 -1.875 0 Mabsadd L closepath F
.4419 .23043 -1.875 0 Mabsadd m
.4419 .23043 0 2.5 Mabsadd L
.4419 .23043 1.875 0 Mabsadd L
.4419 .23043 0 -2.5 Mabsadd L
.4419 .23043 -1.875 0 Mabsadd L closepath F
.44286 .23 -1.875 0 Mabsadd m
.44286 .23 0 2.5 Mabsadd L
.44286 .23 1.875 0 Mabsadd L
.44286 .23 0 -2.5 Mabsadd L
.44286 .23 -1.875 0 Mabsadd L closepath F
.44381 .22842 -1.875 0 Mabsadd m
.44381 .22842 0 2.5 Mabsadd L
.44381 .22842 1.875 0 Mabsadd L
.44381 .22842 0 -2.5 Mabsadd L
.44381 .22842 -1.875 0 Mabsadd L closepath F
.44476 .22163 -1.875 0 Mabsadd m
.44476 .22163 0 2.5 Mabsadd L
.44476 .22163 1.875 0 Mabsadd L
.44476 .22163 0 -2.5 Mabsadd L
.44476 .22163 -1.875 0 Mabsadd L closepath F
.44571 .18759 -1.875 0 Mabsadd m
.44571 .18759 0 2.5 Mabsadd L
.44571 .18759 1.875 0 Mabsadd L
.44571 .18759 0 -2.5 Mabsadd L
.44571 .18759 -1.875 0 Mabsadd L closepath F
.44667 .18758 -1.875 0 Mabsadd m
.44667 .18758 0 2.5 Mabsadd L
.44667 .18758 1.875 0 Mabsadd L
.44667 .18758 0 -2.5 Mabsadd L
.44667 .18758 -1.875 0 Mabsadd L closepath F
.44762 .1891 -1.875 0 Mabsadd m
.44762 .1891 0 2.5 Mabsadd L
.44762 .1891 1.875 0 Mabsadd L
.44762 .1891 0 -2.5 Mabsadd L
.44762 .1891 -1.875 0 Mabsadd L closepath F
.44857 .20253 -1.875 0 Mabsadd m
.44857 .20253 0 2.5 Mabsadd L
.44857 .20253 1.875 0 Mabsadd L
.44857 .20253 0 -2.5 Mabsadd L
.44857 .20253 -1.875 0 Mabsadd L closepath F
.44952 .23097 -1.875 0 Mabsadd m
.44952 .23097 0 2.5 Mabsadd L
.44952 .23097 1.875 0 Mabsadd L
.44952 .23097 0 -2.5 Mabsadd L
.44952 .23097 -1.875 0 Mabsadd L closepath F
.45048 .23532 -1.875 0 Mabsadd m
.45048 .23532 0 2.5 Mabsadd L
.45048 .23532 1.875 0 Mabsadd L
.45048 .23532 0 -2.5 Mabsadd L
.45048 .23532 -1.875 0 Mabsadd L closepath F
.45143 .0632 -1.875 0 Mabsadd m
.45143 .0632 0 2.5 Mabsadd L
.45143 .0632 1.875 0 Mabsadd L
.45143 .0632 0 -2.5 Mabsadd L
.45143 .0632 -1.875 0 Mabsadd L closepath F
.45238 .27164 -1.875 0 Mabsadd m
.45238 .27164 0 2.5 Mabsadd L
.45238 .27164 1.875 0 Mabsadd L
.45238 .27164 0 -2.5 Mabsadd L
.45238 .27164 -1.875 0 Mabsadd L closepath F
.45333 .26013 -1.875 0 Mabsadd m
.45333 .26013 0 2.5 Mabsadd L
.45333 .26013 1.875 0 Mabsadd L
.45333 .26013 0 -2.5 Mabsadd L
.45333 .26013 -1.875 0 Mabsadd L closepath F
.45429 .13373 -1.875 0 Mabsadd m
.45429 .13373 0 2.5 Mabsadd L
.45429 .13373 1.875 0 Mabsadd L
.45429 .13373 0 -2.5 Mabsadd L
.45429 .13373 -1.875 0 Mabsadd L closepath F
.45524 .14421 -1.875 0 Mabsadd m
.45524 .14421 0 2.5 Mabsadd L
.45524 .14421 1.875 0 Mabsadd L
.45524 .14421 0 -2.5 Mabsadd L
.45524 .14421 -1.875 0 Mabsadd L closepath F
.45619 .14529 -1.875 0 Mabsadd m
.45619 .14529 0 2.5 Mabsadd L
.45619 .14529 1.875 0 Mabsadd L
.45619 .14529 0 -2.5 Mabsadd L
.45619 .14529 -1.875 0 Mabsadd L closepath F
.45714 .15124 -1.875 0 Mabsadd m
.45714 .15124 0 2.5 Mabsadd L
.45714 .15124 1.875 0 Mabsadd L
.45714 .15124 0 -2.5 Mabsadd L
.45714 .15124 -1.875 0 Mabsadd L closepath F
.4581 .15127 -1.875 0 Mabsadd m
.4581 .15127 0 2.5 Mabsadd L
.4581 .15127 1.875 0 Mabsadd L
.4581 .15127 0 -2.5 Mabsadd L
.4581 .15127 -1.875 0 Mabsadd L closepath F
.45905 .15135 -1.875 0 Mabsadd m
.45905 .15135 0 2.5 Mabsadd L
.45905 .15135 1.875 0 Mabsadd L
.45905 .15135 0 -2.5 Mabsadd L
.45905 .15135 -1.875 0 Mabsadd L closepath F
.46 .15143 -1.875 0 Mabsadd m
.46 .15143 0 2.5 Mabsadd L
.46 .15143 1.875 0 Mabsadd L
.46 .15143 0 -2.5 Mabsadd L
.46 .15143 -1.875 0 Mabsadd L closepath F
.46095 .15149 -1.875 0 Mabsadd m
.46095 .15149 0 2.5 Mabsadd L
.46095 .15149 1.875 0 Mabsadd L
.46095 .15149 0 -2.5 Mabsadd L
.46095 .15149 -1.875 0 Mabsadd L closepath F
.4619 .15152 -1.875 0 Mabsadd m
.4619 .15152 0 2.5 Mabsadd L
.4619 .15152 1.875 0 Mabsadd L
.4619 .15152 0 -2.5 Mabsadd L
.4619 .15152 -1.875 0 Mabsadd L closepath F
.46286 .15152 -1.875 0 Mabsadd m
.46286 .15152 0 2.5 Mabsadd L
.46286 .15152 1.875 0 Mabsadd L
.46286 .15152 0 -2.5 Mabsadd L
.46286 .15152 -1.875 0 Mabsadd L closepath F
.46381 .15152 -1.875 0 Mabsadd m
.46381 .15152 0 2.5 Mabsadd L
.46381 .15152 1.875 0 Mabsadd L
.46381 .15152 0 -2.5 Mabsadd L
.46381 .15152 -1.875 0 Mabsadd L closepath F
.46476 .15151 -1.875 0 Mabsadd m
.46476 .15151 0 2.5 Mabsadd L
.46476 .15151 1.875 0 Mabsadd L
.46476 .15151 0 -2.5 Mabsadd L
.46476 .15151 -1.875 0 Mabsadd L closepath F
.46571 .1515 -1.875 0 Mabsadd m
.46571 .1515 0 2.5 Mabsadd L
.46571 .1515 1.875 0 Mabsadd L
.46571 .1515 0 -2.5 Mabsadd L
.46571 .1515 -1.875 0 Mabsadd L closepath F
.46667 .15149 -1.875 0 Mabsadd m
.46667 .15149 0 2.5 Mabsadd L
.46667 .15149 1.875 0 Mabsadd L
.46667 .15149 0 -2.5 Mabsadd L
.46667 .15149 -1.875 0 Mabsadd L closepath F
.46762 .15148 -1.875 0 Mabsadd m
.46762 .15148 0 2.5 Mabsadd L
.46762 .15148 1.875 0 Mabsadd L
.46762 .15148 0 -2.5 Mabsadd L
.46762 .15148 -1.875 0 Mabsadd L closepath F
.46857 .15149 -1.875 0 Mabsadd m
.46857 .15149 0 2.5 Mabsadd L
.46857 .15149 1.875 0 Mabsadd L
.46857 .15149 0 -2.5 Mabsadd L
.46857 .15149 -1.875 0 Mabsadd L closepath F
.46952 .15149 -1.875 0 Mabsadd m
.46952 .15149 0 2.5 Mabsadd L
.46952 .15149 1.875 0 Mabsadd L
.46952 .15149 0 -2.5 Mabsadd L
.46952 .15149 -1.875 0 Mabsadd L closepath F
.47048 .1515 -1.875 0 Mabsadd m
.47048 .1515 0 2.5 Mabsadd L
.47048 .1515 1.875 0 Mabsadd L
.47048 .1515 0 -2.5 Mabsadd L
.47048 .1515 -1.875 0 Mabsadd L closepath F
.47143 .1515 -1.875 0 Mabsadd m
.47143 .1515 0 2.5 Mabsadd L
.47143 .1515 1.875 0 Mabsadd L
.47143 .1515 0 -2.5 Mabsadd L
.47143 .1515 -1.875 0 Mabsadd L closepath F
.47238 .1515 -1.875 0 Mabsadd m
.47238 .1515 0 2.5 Mabsadd L
.47238 .1515 1.875 0 Mabsadd L
.47238 .1515 0 -2.5 Mabsadd L
.47238 .1515 -1.875 0 Mabsadd L closepath F
.47333 .15151 -1.875 0 Mabsadd m
.47333 .15151 0 2.5 Mabsadd L
.47333 .15151 1.875 0 Mabsadd L
.47333 .15151 0 -2.5 Mabsadd L
.47333 .15151 -1.875 0 Mabsadd L closepath F
.47429 .15151 -1.875 0 Mabsadd m
.47429 .15151 0 2.5 Mabsadd L
.47429 .15151 1.875 0 Mabsadd L
.47429 .15151 0 -2.5 Mabsadd L
.47429 .15151 -1.875 0 Mabsadd L closepath F
.47524 .15151 -1.875 0 Mabsadd m
.47524 .15151 0 2.5 Mabsadd L
.47524 .15151 1.875 0 Mabsadd L
.47524 .15151 0 -2.5 Mabsadd L
.47524 .15151 -1.875 0 Mabsadd L closepath F
.47619 .15151 -1.875 0 Mabsadd m
.47619 .15151 0 2.5 Mabsadd L
.47619 .15151 1.875 0 Mabsadd L
.47619 .15151 0 -2.5 Mabsadd L
.47619 .15151 -1.875 0 Mabsadd L closepath F
.47714 .15151 -1.875 0 Mabsadd m
.47714 .15151 0 2.5 Mabsadd L
.47714 .15151 1.875 0 Mabsadd L
.47714 .15151 0 -2.5 Mabsadd L
.47714 .15151 -1.875 0 Mabsadd L closepath F
.4781 .15151 -1.875 0 Mabsadd m
.4781 .15151 0 2.5 Mabsadd L
.4781 .15151 1.875 0 Mabsadd L
.4781 .15151 0 -2.5 Mabsadd L
.4781 .15151 -1.875 0 Mabsadd L closepath F
.47905 .15151 -1.875 0 Mabsadd m
.47905 .15151 0 2.5 Mabsadd L
.47905 .15151 1.875 0 Mabsadd L
.47905 .15151 0 -2.5 Mabsadd L
.47905 .15151 -1.875 0 Mabsadd L closepath F
.48 .15151 -1.875 0 Mabsadd m
.48 .15151 0 2.5 Mabsadd L
.48 .15151 1.875 0 Mabsadd L
.48 .15151 0 -2.5 Mabsadd L
.48 .15151 -1.875 0 Mabsadd L closepath F
.48095 .15151 -1.875 0 Mabsadd m
.48095 .15151 0 2.5 Mabsadd L
.48095 .15151 1.875 0 Mabsadd L
.48095 .15151 0 -2.5 Mabsadd L
.48095 .15151 -1.875 0 Mabsadd L closepath F
.4819 .15151 -1.875 0 Mabsadd m
.4819 .15151 0 2.5 Mabsadd L
.4819 .15151 1.875 0 Mabsadd L
.4819 .15151 0 -2.5 Mabsadd L
.4819 .15151 -1.875 0 Mabsadd L closepath F
.48286 .15151 -1.875 0 Mabsadd m
.48286 .15151 0 2.5 Mabsadd L
.48286 .15151 1.875 0 Mabsadd L
.48286 .15151 0 -2.5 Mabsadd L
.48286 .15151 -1.875 0 Mabsadd L closepath F
.48381 .15151 -1.875 0 Mabsadd m
.48381 .15151 0 2.5 Mabsadd L
.48381 .15151 1.875 0 Mabsadd L
.48381 .15151 0 -2.5 Mabsadd L
.48381 .15151 -1.875 0 Mabsadd L closepath F
.48476 .1515 -1.875 0 Mabsadd m
.48476 .1515 0 2.5 Mabsadd L
.48476 .1515 1.875 0 Mabsadd L
.48476 .1515 0 -2.5 Mabsadd L
.48476 .1515 -1.875 0 Mabsadd L closepath F
.48571 .1515 -1.875 0 Mabsadd m
.48571 .1515 0 2.5 Mabsadd L
.48571 .1515 1.875 0 Mabsadd L
.48571 .1515 0 -2.5 Mabsadd L
.48571 .1515 -1.875 0 Mabsadd L closepath F
.48667 .1515 -1.875 0 Mabsadd m
.48667 .1515 0 2.5 Mabsadd L
.48667 .1515 1.875 0 Mabsadd L
.48667 .1515 0 -2.5 Mabsadd L
.48667 .1515 -1.875 0 Mabsadd L closepath F
.48762 .1515 -1.875 0 Mabsadd m
.48762 .1515 0 2.5 Mabsadd L
.48762 .1515 1.875 0 Mabsadd L
.48762 .1515 0 -2.5 Mabsadd L
.48762 .1515 -1.875 0 Mabsadd L closepath F
.48857 .1515 -1.875 0 Mabsadd m
.48857 .1515 0 2.5 Mabsadd L
.48857 .1515 1.875 0 Mabsadd L
.48857 .1515 0 -2.5 Mabsadd L
.48857 .1515 -1.875 0 Mabsadd L closepath F
.48952 .15149 -1.875 0 Mabsadd m
.48952 .15149 0 2.5 Mabsadd L
.48952 .15149 1.875 0 Mabsadd L
.48952 .15149 0 -2.5 Mabsadd L
.48952 .15149 -1.875 0 Mabsadd L closepath F
.49048 .15147 -1.875 0 Mabsadd m
.49048 .15147 0 2.5 Mabsadd L
.49048 .15147 1.875 0 Mabsadd L
.49048 .15147 0 -2.5 Mabsadd L
.49048 .15147 -1.875 0 Mabsadd L closepath F
.49143 .15147 -1.875 0 Mabsadd m
.49143 .15147 0 2.5 Mabsadd L
.49143 .15147 1.875 0 Mabsadd L
.49143 .15147 0 -2.5 Mabsadd L
.49143 .15147 -1.875 0 Mabsadd L closepath F
.49238 .15143 -1.875 0 Mabsadd m
.49238 .15143 0 2.5 Mabsadd L
.49238 .15143 1.875 0 Mabsadd L
.49238 .15143 0 -2.5 Mabsadd L
.49238 .15143 -1.875 0 Mabsadd L closepath F
.49333 .15142 -1.875 0 Mabsadd m
.49333 .15142 0 2.5 Mabsadd L
.49333 .15142 1.875 0 Mabsadd L
.49333 .15142 0 -2.5 Mabsadd L
.49333 .15142 -1.875 0 Mabsadd L closepath F
.49429 .15123 -1.875 0 Mabsadd m
.49429 .15123 0 2.5 Mabsadd L
.49429 .15123 1.875 0 Mabsadd L
.49429 .15123 0 -2.5 Mabsadd L
.49429 .15123 -1.875 0 Mabsadd L closepath F
.49524 .15111 -1.875 0 Mabsadd m
.49524 .15111 0 2.5 Mabsadd L
.49524 .15111 1.875 0 Mabsadd L
.49524 .15111 0 -2.5 Mabsadd L
.49524 .15111 -1.875 0 Mabsadd L closepath F
.49619 .14957 -1.875 0 Mabsadd m
.49619 .14957 0 2.5 Mabsadd L
.49619 .14957 1.875 0 Mabsadd L
.49619 .14957 0 -2.5 Mabsadd L
.49619 .14957 -1.875 0 Mabsadd L closepath F
.49714 .14826 -1.875 0 Mabsadd m
.49714 .14826 0 2.5 Mabsadd L
.49714 .14826 1.875 0 Mabsadd L
.49714 .14826 0 -2.5 Mabsadd L
.49714 .14826 -1.875 0 Mabsadd L closepath F
.4981 .1427 -1.875 0 Mabsadd m
.4981 .1427 0 2.5 Mabsadd L
.4981 .1427 1.875 0 Mabsadd L
.4981 .1427 0 -2.5 Mabsadd L
.4981 .1427 -1.875 0 Mabsadd L closepath F
.49905 .11531 -1.875 0 Mabsadd m
.49905 .11531 0 2.5 Mabsadd L
.49905 .11531 1.875 0 Mabsadd L
.49905 .11531 0 -2.5 Mabsadd L
.49905 .11531 -1.875 0 Mabsadd L closepath F
.5 .1153 -1.875 0 Mabsadd m
.5 .1153 0 2.5 Mabsadd L
.5 .1153 1.875 0 Mabsadd L
.5 .1153 0 -2.5 Mabsadd L
.5 .1153 -1.875 0 Mabsadd L closepath F
.50095 .11646 -1.875 0 Mabsadd m
.50095 .11646 0 2.5 Mabsadd L
.50095 .11646 1.875 0 Mabsadd L
.50095 .11646 0 -2.5 Mabsadd L
.50095 .11646 -1.875 0 Mabsadd L closepath F
.5019 .12766 -1.875 0 Mabsadd m
.5019 .12766 0 2.5 Mabsadd L
.5019 .12766 1.875 0 Mabsadd L
.5019 .12766 0 -2.5 Mabsadd L
.5019 .12766 -1.875 0 Mabsadd L closepath F
.50286 .14561 -1.875 0 Mabsadd m
.50286 .14561 0 2.5 Mabsadd L
.50286 .14561 1.875 0 Mabsadd L
.50286 .14561 0 -2.5 Mabsadd L
.50286 .14561 -1.875 0 Mabsadd L closepath F
.50381 .62554 -1.875 0 Mabsadd m
.50381 .62554 0 2.5 Mabsadd L
.50381 .62554 1.875 0 Mabsadd L
.50381 .62554 0 -2.5 Mabsadd L
.50381 .62554 -1.875 0 Mabsadd L closepath F
.50476 .49142 -1.875 0 Mabsadd m
.50476 .49142 0 2.5 Mabsadd L
.50476 .49142 1.875 0 Mabsadd L
.50476 .49142 0 -2.5 Mabsadd L
.50476 .49142 -1.875 0 Mabsadd L closepath F
.50571 .69255 -1.875 0 Mabsadd m
.50571 .69255 0 2.5 Mabsadd L
.50571 .69255 1.875 0 Mabsadd L
.50571 .69255 0 -2.5 Mabsadd L
.50571 .69255 -1.875 0 Mabsadd L closepath F
.50667 .6734 -1.875 0 Mabsadd m
.50667 .6734 0 2.5 Mabsadd L
.50667 .6734 1.875 0 Mabsadd L
.50667 .6734 0 -2.5 Mabsadd L
.50667 .6734 -1.875 0 Mabsadd L closepath F
.50762 .41456 -1.875 0 Mabsadd m
.50762 .41456 0 2.5 Mabsadd L
.50762 .41456 1.875 0 Mabsadd L
.50762 .41456 0 -2.5 Mabsadd L
.50762 .41456 -1.875 0 Mabsadd L closepath F
.50857 .42767 -1.875 0 Mabsadd m
.50857 .42767 0 2.5 Mabsadd L
.50857 .42767 1.875 0 Mabsadd L
.50857 .42767 0 -2.5 Mabsadd L
.50857 .42767 -1.875 0 Mabsadd L closepath F
.50952 .42912 -1.875 0 Mabsadd m
.50952 .42912 0 2.5 Mabsadd L
.50952 .42912 1.875 0 Mabsadd L
.50952 .42912 0 -2.5 Mabsadd L
.50952 .42912 -1.875 0 Mabsadd L closepath F
.51048 .4374 -1.875 0 Mabsadd m
.51048 .4374 0 2.5 Mabsadd L
.51048 .4374 1.875 0 Mabsadd L
.51048 .4374 0 -2.5 Mabsadd L
.51048 .4374 -1.875 0 Mabsadd L closepath F
.51143 .43744 -1.875 0 Mabsadd m
.51143 .43744 0 2.5 Mabsadd L
.51143 .43744 1.875 0 Mabsadd L
.51143 .43744 0 -2.5 Mabsadd L
.51143 .43744 -1.875 0 Mabsadd L closepath F
.51238 .43754 -1.875 0 Mabsadd m
.51238 .43754 0 2.5 Mabsadd L
.51238 .43754 1.875 0 Mabsadd L
.51238 .43754 0 -2.5 Mabsadd L
.51238 .43754 -1.875 0 Mabsadd L closepath F
.51333 .43764 -1.875 0 Mabsadd m
.51333 .43764 0 2.5 Mabsadd L
.51333 .43764 1.875 0 Mabsadd L
.51333 .43764 0 -2.5 Mabsadd L
.51333 .43764 -1.875 0 Mabsadd L closepath F
.51429 .43771 -1.875 0 Mabsadd m
.51429 .43771 0 2.5 Mabsadd L
.51429 .43771 1.875 0 Mabsadd L
.51429 .43771 0 -2.5 Mabsadd L
.51429 .43771 -1.875 0 Mabsadd L closepath F
.51524 .43775 -1.875 0 Mabsadd m
.51524 .43775 0 2.5 Mabsadd L
.51524 .43775 1.875 0 Mabsadd L
.51524 .43775 0 -2.5 Mabsadd L
.51524 .43775 -1.875 0 Mabsadd L closepath F
.51619 .43776 -1.875 0 Mabsadd m
.51619 .43776 0 2.5 Mabsadd L
.51619 .43776 1.875 0 Mabsadd L
.51619 .43776 0 -2.5 Mabsadd L
.51619 .43776 -1.875 0 Mabsadd L closepath F
.51714 .43775 -1.875 0 Mabsadd m
.51714 .43775 0 2.5 Mabsadd L
.51714 .43775 1.875 0 Mabsadd L
.51714 .43775 0 -2.5 Mabsadd L
.51714 .43775 -1.875 0 Mabsadd L closepath F
.5181 .43774 -1.875 0 Mabsadd m
.5181 .43774 0 2.5 Mabsadd L
.5181 .43774 1.875 0 Mabsadd L
.5181 .43774 0 -2.5 Mabsadd L
.5181 .43774 -1.875 0 Mabsadd L closepath F
.51905 .43773 -1.875 0 Mabsadd m
.51905 .43773 0 2.5 Mabsadd L
.51905 .43773 1.875 0 Mabsadd L
.51905 .43773 0 -2.5 Mabsadd L
.51905 .43773 -1.875 0 Mabsadd L closepath F
.52 .43773 -1.875 0 Mabsadd m
.52 .43773 0 2.5 Mabsadd L
.52 .43773 1.875 0 Mabsadd L
.52 .43773 0 -2.5 Mabsadd L
.52 .43773 -1.875 0 Mabsadd L closepath F
.52095 .43776 -1.875 0 Mabsadd m
.52095 .43776 0 2.5 Mabsadd L
.52095 .43776 1.875 0 Mabsadd L
.52095 .43776 0 -2.5 Mabsadd L
.52095 .43776 -1.875 0 Mabsadd L closepath F
.5219 .43779 -1.875 0 Mabsadd m
.5219 .43779 0 2.5 Mabsadd L
.5219 .43779 1.875 0 Mabsadd L
.5219 .43779 0 -2.5 Mabsadd L
.5219 .43779 -1.875 0 Mabsadd L closepath F
.52286 .43779 -1.875 0 Mabsadd m
.52286 .43779 0 2.5 Mabsadd L
.52286 .43779 1.875 0 Mabsadd L
.52286 .43779 0 -2.5 Mabsadd L
.52286 .43779 -1.875 0 Mabsadd L closepath F
.52381 .43779 -1.875 0 Mabsadd m
.52381 .43779 0 2.5 Mabsadd L
.52381 .43779 1.875 0 Mabsadd L
.52381 .43779 0 -2.5 Mabsadd L
.52381 .43779 -1.875 0 Mabsadd L closepath F
.52476 .4378 -1.875 0 Mabsadd m
.52476 .4378 0 2.5 Mabsadd L
.52476 .4378 1.875 0 Mabsadd L
.52476 .4378 0 -2.5 Mabsadd L
.52476 .4378 -1.875 0 Mabsadd L closepath F
.52571 .4378 -1.875 0 Mabsadd m
.52571 .4378 0 2.5 Mabsadd L
.52571 .4378 1.875 0 Mabsadd L
.52571 .4378 0 -2.5 Mabsadd L
.52571 .4378 -1.875 0 Mabsadd L closepath F
.52667 .4378 -1.875 0 Mabsadd m
.52667 .4378 0 2.5 Mabsadd L
.52667 .4378 1.875 0 Mabsadd L
.52667 .4378 0 -2.5 Mabsadd L
.52667 .4378 -1.875 0 Mabsadd L closepath F
.52762 .4378 -1.875 0 Mabsadd m
.52762 .4378 0 2.5 Mabsadd L
.52762 .4378 1.875 0 Mabsadd L
.52762 .4378 0 -2.5 Mabsadd L
.52762 .4378 -1.875 0 Mabsadd L closepath F
.52857 .43781 -1.875 0 Mabsadd m
.52857 .43781 0 2.5 Mabsadd L
.52857 .43781 1.875 0 Mabsadd L
.52857 .43781 0 -2.5 Mabsadd L
.52857 .43781 -1.875 0 Mabsadd L closepath F
.52952 .43781 -1.875 0 Mabsadd m
.52952 .43781 0 2.5 Mabsadd L
.52952 .43781 1.875 0 Mabsadd L
.52952 .43781 0 -2.5 Mabsadd L
.52952 .43781 -1.875 0 Mabsadd L closepath F
.53048 .43781 -1.875 0 Mabsadd m
.53048 .43781 0 2.5 Mabsadd L
.53048 .43781 1.875 0 Mabsadd L
.53048 .43781 0 -2.5 Mabsadd L
.53048 .43781 -1.875 0 Mabsadd L closepath F
.53143 .43781 -1.875 0 Mabsadd m
.53143 .43781 0 2.5 Mabsadd L
.53143 .43781 1.875 0 Mabsadd L
.53143 .43781 0 -2.5 Mabsadd L
.53143 .43781 -1.875 0 Mabsadd L closepath F
.53238 .43781 -1.875 0 Mabsadd m
.53238 .43781 0 2.5 Mabsadd L
.53238 .43781 1.875 0 Mabsadd L
.53238 .43781 0 -2.5 Mabsadd L
.53238 .43781 -1.875 0 Mabsadd L closepath F
.53333 .43781 -1.875 0 Mabsadd m
.53333 .43781 0 2.5 Mabsadd L
.53333 .43781 1.875 0 Mabsadd L
.53333 .43781 0 -2.5 Mabsadd L
.53333 .43781 -1.875 0 Mabsadd L closepath F
.53429 .43781 -1.875 0 Mabsadd m
.53429 .43781 0 2.5 Mabsadd L
.53429 .43781 1.875 0 Mabsadd L
.53429 .43781 0 -2.5 Mabsadd L
.53429 .43781 -1.875 0 Mabsadd L closepath F
.53524 .43781 -1.875 0 Mabsadd m
.53524 .43781 0 2.5 Mabsadd L
.53524 .43781 1.875 0 Mabsadd L
.53524 .43781 0 -2.5 Mabsadd L
.53524 .43781 -1.875 0 Mabsadd L closepath F
.53619 .43781 -1.875 0 Mabsadd m
.53619 .43781 0 2.5 Mabsadd L
.53619 .43781 1.875 0 Mabsadd L
.53619 .43781 0 -2.5 Mabsadd L
.53619 .43781 -1.875 0 Mabsadd L closepath F
.53714 .43781 -1.875 0 Mabsadd m
.53714 .43781 0 2.5 Mabsadd L
.53714 .43781 1.875 0 Mabsadd L
.53714 .43781 0 -2.5 Mabsadd L
.53714 .43781 -1.875 0 Mabsadd L closepath F
.5381 .4378 -1.875 0 Mabsadd m
.5381 .4378 0 2.5 Mabsadd L
.5381 .4378 1.875 0 Mabsadd L
.5381 .4378 0 -2.5 Mabsadd L
.5381 .4378 -1.875 0 Mabsadd L closepath F
.53905 .4378 -1.875 0 Mabsadd m
.53905 .4378 0 2.5 Mabsadd L
.53905 .4378 1.875 0 Mabsadd L
.53905 .4378 0 -2.5 Mabsadd L
.53905 .4378 -1.875 0 Mabsadd L closepath F
.54 .4378 -1.875 0 Mabsadd m
.54 .4378 0 2.5 Mabsadd L
.54 .4378 1.875 0 Mabsadd L
.54 .4378 0 -2.5 Mabsadd L
.54 .4378 -1.875 0 Mabsadd L closepath F
.54095 .4378 -1.875 0 Mabsadd m
.54095 .4378 0 2.5 Mabsadd L
.54095 .4378 1.875 0 Mabsadd L
.54095 .4378 0 -2.5 Mabsadd L
.54095 .4378 -1.875 0 Mabsadd L closepath F
.5419 .43779 -1.875 0 Mabsadd m
.5419 .43779 0 2.5 Mabsadd L
.5419 .43779 1.875 0 Mabsadd L
.5419 .43779 0 -2.5 Mabsadd L
.5419 .43779 -1.875 0 Mabsadd L closepath F
.54286 .43779 -1.875 0 Mabsadd m
.54286 .43779 0 2.5 Mabsadd L
.54286 .43779 1.875 0 Mabsadd L
.54286 .43779 0 -2.5 Mabsadd L
.54286 .43779 -1.875 0 Mabsadd L closepath F
.54381 .43778 -1.875 0 Mabsadd m
.54381 .43778 0 2.5 Mabsadd L
.54381 .43778 1.875 0 Mabsadd L
.54381 .43778 0 -2.5 Mabsadd L
.54381 .43778 -1.875 0 Mabsadd L closepath F
.54476 .43775 -1.875 0 Mabsadd m
.54476 .43775 0 2.5 Mabsadd L
.54476 .43775 1.875 0 Mabsadd L
.54476 .43775 0 -2.5 Mabsadd L
.54476 .43775 -1.875 0 Mabsadd L closepath F
.54571 .43774 -1.875 0 Mabsadd m
.54571 .43774 0 2.5 Mabsadd L
.54571 .43774 1.875 0 Mabsadd L
.54571 .43774 0 -2.5 Mabsadd L
.54571 .43774 -1.875 0 Mabsadd L closepath F
.54667 .43767 -1.875 0 Mabsadd m
.54667 .43767 0 2.5 Mabsadd L
.54667 .43767 1.875 0 Mabsadd L
.54667 .43767 0 -2.5 Mabsadd L
.54667 .43767 -1.875 0 Mabsadd L closepath F
.54762 .43763 -1.875 0 Mabsadd m
.54762 .43763 0 2.5 Mabsadd L
.54762 .43763 1.875 0 Mabsadd L
.54762 .43763 0 -2.5 Mabsadd L
.54762 .43763 -1.875 0 Mabsadd L closepath F
.54857 .43719 -1.875 0 Mabsadd m
.54857 .43719 0 2.5 Mabsadd L
.54857 .43719 1.875 0 Mabsadd L
.54857 .43719 0 -2.5 Mabsadd L
.54857 .43719 -1.875 0 Mabsadd L closepath F
.54952 .43688 -1.875 0 Mabsadd m
.54952 .43688 0 2.5 Mabsadd L
.54952 .43688 1.875 0 Mabsadd L
.54952 .43688 0 -2.5 Mabsadd L
.54952 .43688 -1.875 0 Mabsadd L closepath F
.55048 .43577 -1.875 0 Mabsadd m
.55048 .43577 0 2.5 Mabsadd L
.55048 .43577 1.875 0 Mabsadd L
.55048 .43577 0 -2.5 Mabsadd L
.55048 .43577 -1.875 0 Mabsadd L closepath F
.55143 .43118 -1.875 0 Mabsadd m
.55143 .43118 0 2.5 Mabsadd L
.55143 .43118 1.875 0 Mabsadd L
.55143 .43118 0 -2.5 Mabsadd L
.55143 .43118 -1.875 0 Mabsadd L closepath F
.55238 .40911 -1.875 0 Mabsadd m
.55238 .40911 0 2.5 Mabsadd L
.55238 .40911 1.875 0 Mabsadd L
.55238 .40911 0 -2.5 Mabsadd L
.55238 .40911 -1.875 0 Mabsadd L closepath F
.55333 .40909 -1.875 0 Mabsadd m
.55333 .40909 0 2.5 Mabsadd L
.55333 .40909 1.875 0 Mabsadd L
.55333 .40909 0 -2.5 Mabsadd L
.55333 .40909 -1.875 0 Mabsadd L closepath F
.55429 .40997 -1.875 0 Mabsadd m
.55429 .40997 0 2.5 Mabsadd L
.55429 .40997 1.875 0 Mabsadd L
.55429 .40997 0 -2.5 Mabsadd L
.55429 .40997 -1.875 0 Mabsadd L closepath F
.55524 .4192 -1.875 0 Mabsadd m
.55524 .4192 0 2.5 Mabsadd L
.55524 .4192 1.875 0 Mabsadd L
.55524 .4192 0 -2.5 Mabsadd L
.55524 .4192 -1.875 0 Mabsadd L closepath F
.55619 .43072 -1.875 0 Mabsadd m
.55619 .43072 0 2.5 Mabsadd L
.55619 .43072 1.875 0 Mabsadd L
.55619 .43072 0 -2.5 Mabsadd L
.55619 .43072 -1.875 0 Mabsadd L closepath F
.55714 .41115 -1.875 0 Mabsadd m
.55714 .41115 0 2.5 Mabsadd L
.55714 .41115 1.875 0 Mabsadd L
.55714 .41115 0 -2.5 Mabsadd L
.55714 .41115 -1.875 0 Mabsadd L closepath F
.5581 .32782 -1.875 0 Mabsadd m
.5581 .32782 0 2.5 Mabsadd L
.5581 .32782 1.875 0 Mabsadd L
.5581 .32782 0 -2.5 Mabsadd L
.5581 .32782 -1.875 0 Mabsadd L closepath F
.55905 .48119 -1.875 0 Mabsadd m
.55905 .48119 0 2.5 Mabsadd L
.55905 .48119 1.875 0 Mabsadd L
.55905 .48119 0 -2.5 Mabsadd L
.55905 .48119 -1.875 0 Mabsadd L closepath F
.56 .46467 -1.875 0 Mabsadd m
.56 .46467 0 2.5 Mabsadd L
.56 .46467 1.875 0 Mabsadd L
.56 .46467 0 -2.5 Mabsadd L
.56 .46467 -1.875 0 Mabsadd L closepath F
.56095 .06018 -1.875 0 Mabsadd m
.56095 .06018 0 2.5 Mabsadd L
.56095 .06018 1.875 0 Mabsadd L
.56095 .06018 0 -2.5 Mabsadd L
.56095 .06018 -1.875 0 Mabsadd L closepath F
.5619 .04471 -1.875 0 Mabsadd m
.5619 .04471 0 2.5 Mabsadd L
.5619 .04471 1.875 0 Mabsadd L
.5619 .04471 0 -2.5 Mabsadd L
.5619 .04471 -1.875 0 Mabsadd L closepath F
.56286 .04275 -1.875 0 Mabsadd m
.56286 .04275 0 2.5 Mabsadd L
.56286 .04275 1.875 0 Mabsadd L
.56286 .04275 0 -2.5 Mabsadd L
.56286 .04275 -1.875 0 Mabsadd L closepath F
.56381 .03116 -1.875 0 Mabsadd m
.56381 .03116 0 2.5 Mabsadd L
.56381 .03116 1.875 0 Mabsadd L
.56381 .03116 0 -2.5 Mabsadd L
.56381 .03116 -1.875 0 Mabsadd L closepath F
.56476 .03111 -1.875 0 Mabsadd m
.56476 .03111 0 2.5 Mabsadd L
.56476 .03111 1.875 0 Mabsadd L
.56476 .03111 0 -2.5 Mabsadd L
.56476 .03111 -1.875 0 Mabsadd L closepath F
.56571 .03099 -1.875 0 Mabsadd m
.56571 .03099 0 2.5 Mabsadd L
.56571 .03099 1.875 0 Mabsadd L
.56571 .03099 0 -2.5 Mabsadd L
.56571 .03099 -1.875 0 Mabsadd L closepath F
.56667 .03085 -1.875 0 Mabsadd m
.56667 .03085 0 2.5 Mabsadd L
.56667 .03085 1.875 0 Mabsadd L
.56667 .03085 0 -2.5 Mabsadd L
.56667 .03085 -1.875 0 Mabsadd L closepath F
.56762 .03077 -1.875 0 Mabsadd m
.56762 .03077 0 2.5 Mabsadd L
.56762 .03077 1.875 0 Mabsadd L
.56762 .03077 0 -2.5 Mabsadd L
.56762 .03077 -1.875 0 Mabsadd L closepath F
.56857 .03073 -1.875 0 Mabsadd m
.56857 .03073 0 2.5 Mabsadd L
.56857 .03073 1.875 0 Mabsadd L
.56857 .03073 0 -2.5 Mabsadd L
.56857 .03073 -1.875 0 Mabsadd L closepath F
.56952 .03072 -1.875 0 Mabsadd m
.56952 .03072 0 2.5 Mabsadd L
.56952 .03072 1.875 0 Mabsadd L
.56952 .03072 0 -2.5 Mabsadd L
.56952 .03072 -1.875 0 Mabsadd L closepath F
.57048 .03072 -1.875 0 Mabsadd m
.57048 .03072 0 2.5 Mabsadd L
.57048 .03072 1.875 0 Mabsadd L
.57048 .03072 0 -2.5 Mabsadd L
.57048 .03072 -1.875 0 Mabsadd L closepath F
.57143 .03069 -1.875 0 Mabsadd m
.57143 .03069 0 2.5 Mabsadd L
.57143 .03069 1.875 0 Mabsadd L
.57143 .03069 0 -2.5 Mabsadd L
.57143 .03069 -1.875 0 Mabsadd L closepath F
.57238 .03059 -1.875 0 Mabsadd m
.57238 .03059 0 2.5 Mabsadd L
.57238 .03059 1.875 0 Mabsadd L
.57238 .03059 0 -2.5 Mabsadd L
.57238 .03059 -1.875 0 Mabsadd L closepath F
.57333 .03058 -1.875 0 Mabsadd m
.57333 .03058 0 2.5 Mabsadd L
.57333 .03058 1.875 0 Mabsadd L
.57333 .03058 0 -2.5 Mabsadd L
.57333 .03058 -1.875 0 Mabsadd L closepath F
.57429 .03056 -1.875 0 Mabsadd m
.57429 .03056 0 2.5 Mabsadd L
.57429 .03056 1.875 0 Mabsadd L
.57429 .03056 0 -2.5 Mabsadd L
.57429 .03056 -1.875 0 Mabsadd L closepath F
.57524 .03056 -1.875 0 Mabsadd m
.57524 .03056 0 2.5 Mabsadd L
.57524 .03056 1.875 0 Mabsadd L
.57524 .03056 0 -2.5 Mabsadd L
.57524 .03056 -1.875 0 Mabsadd L closepath F
.57619 .03055 -1.875 0 Mabsadd m
.57619 .03055 0 2.5 Mabsadd L
.57619 .03055 1.875 0 Mabsadd L
.57619 .03055 0 -2.5 Mabsadd L
.57619 .03055 -1.875 0 Mabsadd L closepath F
.57714 .03054 -1.875 0 Mabsadd m
.57714 .03054 0 2.5 Mabsadd L
.57714 .03054 1.875 0 Mabsadd L
.57714 .03054 0 -2.5 Mabsadd L
.57714 .03054 -1.875 0 Mabsadd L closepath F
.5781 .03054 -1.875 0 Mabsadd m
.5781 .03054 0 2.5 Mabsadd L
.5781 .03054 1.875 0 Mabsadd L
.5781 .03054 0 -2.5 Mabsadd L
.5781 .03054 -1.875 0 Mabsadd L closepath F
.57905 .03054 -1.875 0 Mabsadd m
.57905 .03054 0 2.5 Mabsadd L
.57905 .03054 1.875 0 Mabsadd L
.57905 .03054 0 -2.5 Mabsadd L
.57905 .03054 -1.875 0 Mabsadd L closepath F
.58 .03054 -1.875 0 Mabsadd m
.58 .03054 0 2.5 Mabsadd L
.58 .03054 1.875 0 Mabsadd L
.58 .03054 0 -2.5 Mabsadd L
.58 .03054 -1.875 0 Mabsadd L closepath F
.58095 .03053 -1.875 0 Mabsadd m
.58095 .03053 0 2.5 Mabsadd L
.58095 .03053 1.875 0 Mabsadd L
.58095 .03053 0 -2.5 Mabsadd L
.58095 .03053 -1.875 0 Mabsadd L closepath F
.5819 .03053 -1.875 0 Mabsadd m
.5819 .03053 0 2.5 Mabsadd L
.5819 .03053 1.875 0 Mabsadd L
.5819 .03053 0 -2.5 Mabsadd L
.5819 .03053 -1.875 0 Mabsadd L closepath F
.58286 .03053 -1.875 0 Mabsadd m
.58286 .03053 0 2.5 Mabsadd L
.58286 .03053 1.875 0 Mabsadd L
.58286 .03053 0 -2.5 Mabsadd L
.58286 .03053 -1.875 0 Mabsadd L closepath F
.58381 .03053 -1.875 0 Mabsadd m
.58381 .03053 0 2.5 Mabsadd L
.58381 .03053 1.875 0 Mabsadd L
.58381 .03053 0 -2.5 Mabsadd L
.58381 .03053 -1.875 0 Mabsadd L closepath F
.58476 .03053 -1.875 0 Mabsadd m
.58476 .03053 0 2.5 Mabsadd L
.58476 .03053 1.875 0 Mabsadd L
.58476 .03053 0 -2.5 Mabsadd L
.58476 .03053 -1.875 0 Mabsadd L closepath F
.58571 .03053 -1.875 0 Mabsadd m
.58571 .03053 0 2.5 Mabsadd L
.58571 .03053 1.875 0 Mabsadd L
.58571 .03053 0 -2.5 Mabsadd L
.58571 .03053 -1.875 0 Mabsadd L closepath F
.58667 .03053 -1.875 0 Mabsadd m
.58667 .03053 0 2.5 Mabsadd L
.58667 .03053 1.875 0 Mabsadd L
.58667 .03053 0 -2.5 Mabsadd L
.58667 .03053 -1.875 0 Mabsadd L closepath F
.58762 .03053 -1.875 0 Mabsadd m
.58762 .03053 0 2.5 Mabsadd L
.58762 .03053 1.875 0 Mabsadd L
.58762 .03053 0 -2.5 Mabsadd L
.58762 .03053 -1.875 0 Mabsadd L closepath F
.58857 .03053 -1.875 0 Mabsadd m
.58857 .03053 0 2.5 Mabsadd L
.58857 .03053 1.875 0 Mabsadd L
.58857 .03053 0 -2.5 Mabsadd L
.58857 .03053 -1.875 0 Mabsadd L closepath F
.58952 .03053 -1.875 0 Mabsadd m
.58952 .03053 0 2.5 Mabsadd L
.58952 .03053 1.875 0 Mabsadd L
.58952 .03053 0 -2.5 Mabsadd L
.58952 .03053 -1.875 0 Mabsadd L closepath F
.59048 .03053 -1.875 0 Mabsadd m
.59048 .03053 0 2.5 Mabsadd L
.59048 .03053 1.875 0 Mabsadd L
.59048 .03053 0 -2.5 Mabsadd L
.59048 .03053 -1.875 0 Mabsadd L closepath F
.59143 .03053 -1.875 0 Mabsadd m
.59143 .03053 0 2.5 Mabsadd L
.59143 .03053 1.875 0 Mabsadd L
.59143 .03053 0 -2.5 Mabsadd L
.59143 .03053 -1.875 0 Mabsadd L closepath F
.59238 .03054 -1.875 0 Mabsadd m
.59238 .03054 0 2.5 Mabsadd L
.59238 .03054 1.875 0 Mabsadd L
.59238 .03054 0 -2.5 Mabsadd L
.59238 .03054 -1.875 0 Mabsadd L closepath F
.59333 .03054 -1.875 0 Mabsadd m
.59333 .03054 0 2.5 Mabsadd L
.59333 .03054 1.875 0 Mabsadd L
.59333 .03054 0 -2.5 Mabsadd L
.59333 .03054 -1.875 0 Mabsadd L closepath F
.59429 .03054 -1.875 0 Mabsadd m
.59429 .03054 0 2.5 Mabsadd L
.59429 .03054 1.875 0 Mabsadd L
.59429 .03054 0 -2.5 Mabsadd L
.59429 .03054 -1.875 0 Mabsadd L closepath F
.59524 .03055 -1.875 0 Mabsadd m
.59524 .03055 0 2.5 Mabsadd L
.59524 .03055 1.875 0 Mabsadd L
.59524 .03055 0 -2.5 Mabsadd L
.59524 .03055 -1.875 0 Mabsadd L closepath F
.59619 .03055 -1.875 0 Mabsadd m
.59619 .03055 0 2.5 Mabsadd L
.59619 .03055 1.875 0 Mabsadd L
.59619 .03055 0 -2.5 Mabsadd L
.59619 .03055 -1.875 0 Mabsadd L closepath F
.59714 .03055 -1.875 0 Mabsadd m
.59714 .03055 0 2.5 Mabsadd L
.59714 .03055 1.875 0 Mabsadd L
.59714 .03055 0 -2.5 Mabsadd L
.59714 .03055 -1.875 0 Mabsadd L closepath F
.5981 .03059 -1.875 0 Mabsadd m
.5981 .03059 0 2.5 Mabsadd L
.5981 .03059 1.875 0 Mabsadd L
.5981 .03059 0 -2.5 Mabsadd L
.5981 .03059 -1.875 0 Mabsadd L closepath F
.59905 .03059 -1.875 0 Mabsadd m
.59905 .03059 0 2.5 Mabsadd L
.59905 .03059 1.875 0 Mabsadd L
.59905 .03059 0 -2.5 Mabsadd L
.59905 .03059 -1.875 0 Mabsadd L closepath F
.6 .03066 -1.875 0 Mabsadd m
.6 .03066 0 2.5 Mabsadd L
.6 .03066 1.875 0 Mabsadd L
.6 .03066 0 -2.5 Mabsadd L
.6 .03066 -1.875 0 Mabsadd L closepath F
.60095 .03069 -1.875 0 Mabsadd m
.60095 .03069 0 2.5 Mabsadd L
.60095 .03069 1.875 0 Mabsadd L
.60095 .03069 0 -2.5 Mabsadd L
.60095 .03069 -1.875 0 Mabsadd L closepath F
.6019 .03107 -1.875 0 Mabsadd m
.6019 .03107 0 2.5 Mabsadd L
.6019 .03107 1.875 0 Mabsadd L
.6019 .03107 0 -2.5 Mabsadd L
.6019 .03107 -1.875 0 Mabsadd L closepath F
.60286 .03134 -1.875 0 Mabsadd m
.60286 .03134 0 2.5 Mabsadd L
.60286 .03134 1.875 0 Mabsadd L
.60286 .03134 0 -2.5 Mabsadd L
.60286 .03134 -1.875 0 Mabsadd L closepath F
.60381 .03228 -1.875 0 Mabsadd m
.60381 .03228 0 2.5 Mabsadd L
.60381 .03228 1.875 0 Mabsadd L
.60381 .03228 0 -2.5 Mabsadd L
.60381 .03228 -1.875 0 Mabsadd L closepath F
.60476 .03606 -1.875 0 Mabsadd m
.60476 .03606 0 2.5 Mabsadd L
.60476 .03606 1.875 0 Mabsadd L
.60476 .03606 0 -2.5 Mabsadd L
.60476 .03606 -1.875 0 Mabsadd L closepath F
.60571 .05391 -1.875 0 Mabsadd m
.60571 .05391 0 2.5 Mabsadd L
.60571 .05391 1.875 0 Mabsadd L
.60571 .05391 0 -2.5 Mabsadd L
.60571 .05391 -1.875 0 Mabsadd L closepath F
.60667 .05393 -1.875 0 Mabsadd m
.60667 .05393 0 2.5 Mabsadd L
.60667 .05393 1.875 0 Mabsadd L
.60667 .05393 0 -2.5 Mabsadd L
.60667 .05393 -1.875 0 Mabsadd L closepath F
.60762 .05326 -1.875 0 Mabsadd m
.60762 .05326 0 2.5 Mabsadd L
.60762 .05326 1.875 0 Mabsadd L
.60762 .05326 0 -2.5 Mabsadd L
.60762 .05326 -1.875 0 Mabsadd L closepath F
.60857 .04573 -1.875 0 Mabsadd m
.60857 .04573 0 2.5 Mabsadd L
.60857 .04573 1.875 0 Mabsadd L
.60857 .04573 0 -2.5 Mabsadd L
.60857 .04573 -1.875 0 Mabsadd L closepath F
.60952 .03812 -1.875 0 Mabsadd m
.60952 .03812 0 2.5 Mabsadd L
.60952 .03812 1.875 0 Mabsadd L
.60952 .03812 0 -2.5 Mabsadd L
.60952 .03812 -1.875 0 Mabsadd L closepath F
.61048 .06487 -1.875 0 Mabsadd m
.61048 .06487 0 2.5 Mabsadd L
.61048 .06487 1.875 0 Mabsadd L
.61048 .06487 0 -2.5 Mabsadd L
.61048 .06487 -1.875 0 Mabsadd L closepath F
.61143 .09997 -1.875 0 Mabsadd m
.61143 .09997 0 2.5 Mabsadd L
.61143 .09997 1.875 0 Mabsadd L
.61143 .09997 0 -2.5 Mabsadd L
.61143 .09997 -1.875 0 Mabsadd L closepath F
.61238 .02719 -1.875 0 Mabsadd m
.61238 .02719 0 2.5 Mabsadd L
.61238 .02719 1.875 0 Mabsadd L
.61238 .02719 0 -2.5 Mabsadd L
.61238 .02719 -1.875 0 Mabsadd L closepath F
.61333 .02554 -1.875 0 Mabsadd m
.61333 .02554 0 2.5 Mabsadd L
.61333 .02554 1.875 0 Mabsadd L
.61333 .02554 0 -2.5 Mabsadd L
.61333 .02554 -1.875 0 Mabsadd L closepath F
.61429 .05063 -1.875 0 Mabsadd m
.61429 .05063 0 2.5 Mabsadd L
.61429 .05063 1.875 0 Mabsadd L
.61429 .05063 0 -2.5 Mabsadd L
.61429 .05063 -1.875 0 Mabsadd L closepath F
.61524 .0673 -1.875 0 Mabsadd m
.61524 .0673 0 2.5 Mabsadd L
.61524 .0673 1.875 0 Mabsadd L
.61524 .0673 0 -2.5 Mabsadd L
.61524 .0673 -1.875 0 Mabsadd L closepath F
.61619 .06997 -1.875 0 Mabsadd m
.61619 .06997 0 2.5 Mabsadd L
.61619 .06997 1.875 0 Mabsadd L
.61619 .06997 0 -2.5 Mabsadd L
.61619 .06997 -1.875 0 Mabsadd L closepath F
.61714 .08629 -1.875 0 Mabsadd m
.61714 .08629 0 2.5 Mabsadd L
.61714 .08629 1.875 0 Mabsadd L
.61714 .08629 0 -2.5 Mabsadd L
.61714 .08629 -1.875 0 Mabsadd L closepath F
.6181 .08634 -1.875 0 Mabsadd m
.6181 .08634 0 2.5 Mabsadd L
.6181 .08634 1.875 0 Mabsadd L
.6181 .08634 0 -2.5 Mabsadd L
.6181 .08634 -1.875 0 Mabsadd L closepath F
.61905 .0865 -1.875 0 Mabsadd m
.61905 .0865 0 2.5 Mabsadd L
.61905 .0865 1.875 0 Mabsadd L
.61905 .0865 0 -2.5 Mabsadd L
.61905 .0865 -1.875 0 Mabsadd L closepath F
.62 .08667 -1.875 0 Mabsadd m
.62 .08667 0 2.5 Mabsadd L
.62 .08667 1.875 0 Mabsadd L
.62 .08667 0 -2.5 Mabsadd L
.62 .08667 -1.875 0 Mabsadd L closepath F
.62095 .08678 -1.875 0 Mabsadd m
.62095 .08678 0 2.5 Mabsadd L
.62095 .08678 1.875 0 Mabsadd L
.62095 .08678 0 -2.5 Mabsadd L
.62095 .08678 -1.875 0 Mabsadd L closepath F
.6219 .08683 -1.875 0 Mabsadd m
.6219 .08683 0 2.5 Mabsadd L
.6219 .08683 1.875 0 Mabsadd L
.6219 .08683 0 -2.5 Mabsadd L
.6219 .08683 -1.875 0 Mabsadd L closepath F
.62286 .08685 -1.875 0 Mabsadd m
.62286 .08685 0 2.5 Mabsadd L
.62286 .08685 1.875 0 Mabsadd L
.62286 .08685 0 -2.5 Mabsadd L
.62286 .08685 -1.875 0 Mabsadd L closepath F
.62381 .08694 -1.875 0 Mabsadd m
.62381 .08694 0 2.5 Mabsadd L
.62381 .08694 1.875 0 Mabsadd L
.62381 .08694 0 -2.5 Mabsadd L
.62381 .08694 -1.875 0 Mabsadd L closepath F
.62476 .08721 -1.875 0 Mabsadd m
.62476 .08721 0 2.5 Mabsadd L
.62476 .08721 1.875 0 Mabsadd L
.62476 .08721 0 -2.5 Mabsadd L
.62476 .08721 -1.875 0 Mabsadd L closepath F
.62571 .08723 -1.875 0 Mabsadd m
.62571 .08723 0 2.5 Mabsadd L
.62571 .08723 1.875 0 Mabsadd L
.62571 .08723 0 -2.5 Mabsadd L
.62571 .08723 -1.875 0 Mabsadd L closepath F
.62667 .08728 -1.875 0 Mabsadd m
.62667 .08728 0 2.5 Mabsadd L
.62667 .08728 1.875 0 Mabsadd L
.62667 .08728 0 -2.5 Mabsadd L
.62667 .08728 -1.875 0 Mabsadd L closepath F
.62762 .08728 -1.875 0 Mabsadd m
.62762 .08728 0 2.5 Mabsadd L
.62762 .08728 1.875 0 Mabsadd L
.62762 .08728 0 -2.5 Mabsadd L
.62762 .08728 -1.875 0 Mabsadd L closepath F
.62857 .0873 -1.875 0 Mabsadd m
.62857 .0873 0 2.5 Mabsadd L
.62857 .0873 1.875 0 Mabsadd L
.62857 .0873 0 -2.5 Mabsadd L
.62857 .0873 -1.875 0 Mabsadd L closepath F
.62952 .0873 -1.875 0 Mabsadd m
.62952 .0873 0 2.5 Mabsadd L
.62952 .0873 1.875 0 Mabsadd L
.62952 .0873 0 -2.5 Mabsadd L
.62952 .0873 -1.875 0 Mabsadd L closepath F
.63048 .0873 -1.875 0 Mabsadd m
.63048 .0873 0 2.5 Mabsadd L
.63048 .0873 1.875 0 Mabsadd L
.63048 .0873 0 -2.5 Mabsadd L
.63048 .0873 -1.875 0 Mabsadd L closepath F
.63143 .08731 -1.875 0 Mabsadd m
.63143 .08731 0 2.5 Mabsadd L
.63143 .08731 1.875 0 Mabsadd L
.63143 .08731 0 -2.5 Mabsadd L
.63143 .08731 -1.875 0 Mabsadd L closepath F
.63238 .08731 -1.875 0 Mabsadd m
.63238 .08731 0 2.5 Mabsadd L
.63238 .08731 1.875 0 Mabsadd L
.63238 .08731 0 -2.5 Mabsadd L
.63238 .08731 -1.875 0 Mabsadd L closepath F
.63333 .08731 -1.875 0 Mabsadd m
.63333 .08731 0 2.5 Mabsadd L
.63333 .08731 1.875 0 Mabsadd L
.63333 .08731 0 -2.5 Mabsadd L
.63333 .08731 -1.875 0 Mabsadd L closepath F
.63429 .08731 -1.875 0 Mabsadd m
.63429 .08731 0 2.5 Mabsadd L
.63429 .08731 1.875 0 Mabsadd L
.63429 .08731 0 -2.5 Mabsadd L
.63429 .08731 -1.875 0 Mabsadd L closepath F
.63524 .08731 -1.875 0 Mabsadd m
.63524 .08731 0 2.5 Mabsadd L
.63524 .08731 1.875 0 Mabsadd L
.63524 .08731 0 -2.5 Mabsadd L
.63524 .08731 -1.875 0 Mabsadd L closepath F
.63619 .08731 -1.875 0 Mabsadd m
.63619 .08731 0 2.5 Mabsadd L
.63619 .08731 1.875 0 Mabsadd L
.63619 .08731 0 -2.5 Mabsadd L
.63619 .08731 -1.875 0 Mabsadd L closepath F
.63714 .08731 -1.875 0 Mabsadd m
.63714 .08731 0 2.5 Mabsadd L
.63714 .08731 1.875 0 Mabsadd L
.63714 .08731 0 -2.5 Mabsadd L
.63714 .08731 -1.875 0 Mabsadd L closepath F
.6381 .08731 -1.875 0 Mabsadd m
.6381 .08731 0 2.5 Mabsadd L
.6381 .08731 1.875 0 Mabsadd L
.6381 .08731 0 -2.5 Mabsadd L
.6381 .08731 -1.875 0 Mabsadd L closepath F
.63905 .08731 -1.875 0 Mabsadd m
.63905 .08731 0 2.5 Mabsadd L
.63905 .08731 1.875 0 Mabsadd L
.63905 .08731 0 -2.5 Mabsadd L
.63905 .08731 -1.875 0 Mabsadd L closepath F
.64 .08731 -1.875 0 Mabsadd m
.64 .08731 0 2.5 Mabsadd L
.64 .08731 1.875 0 Mabsadd L
.64 .08731 0 -2.5 Mabsadd L
.64 .08731 -1.875 0 Mabsadd L closepath F
.64095 .08731 -1.875 0 Mabsadd m
.64095 .08731 0 2.5 Mabsadd L
.64095 .08731 1.875 0 Mabsadd L
.64095 .08731 0 -2.5 Mabsadd L
.64095 .08731 -1.875 0 Mabsadd L closepath F
.6419 .08731 -1.875 0 Mabsadd m
.6419 .08731 0 2.5 Mabsadd L
.6419 .08731 1.875 0 Mabsadd L
.6419 .08731 0 -2.5 Mabsadd L
.6419 .08731 -1.875 0 Mabsadd L closepath F
.64286 .08731 -1.875 0 Mabsadd m
.64286 .08731 0 2.5 Mabsadd L
.64286 .08731 1.875 0 Mabsadd L
.64286 .08731 0 -2.5 Mabsadd L
.64286 .08731 -1.875 0 Mabsadd L closepath F
.64381 .08731 -1.875 0 Mabsadd m
.64381 .08731 0 2.5 Mabsadd L
.64381 .08731 1.875 0 Mabsadd L
.64381 .08731 0 -2.5 Mabsadd L
.64381 .08731 -1.875 0 Mabsadd L closepath F
.64476 .08731 -1.875 0 Mabsadd m
.64476 .08731 0 2.5 Mabsadd L
.64476 .08731 1.875 0 Mabsadd L
.64476 .08731 0 -2.5 Mabsadd L
.64476 .08731 -1.875 0 Mabsadd L closepath F
.64571 .08731 -1.875 0 Mabsadd m
.64571 .08731 0 2.5 Mabsadd L
.64571 .08731 1.875 0 Mabsadd L
.64571 .08731 0 -2.5 Mabsadd L
.64571 .08731 -1.875 0 Mabsadd L closepath F
.64667 .08731 -1.875 0 Mabsadd m
.64667 .08731 0 2.5 Mabsadd L
.64667 .08731 1.875 0 Mabsadd L
.64667 .08731 0 -2.5 Mabsadd L
.64667 .08731 -1.875 0 Mabsadd L closepath F
.64762 .08731 -1.875 0 Mabsadd m
.64762 .08731 0 2.5 Mabsadd L
.64762 .08731 1.875 0 Mabsadd L
.64762 .08731 0 -2.5 Mabsadd L
.64762 .08731 -1.875 0 Mabsadd L closepath F
.64857 .0873 -1.875 0 Mabsadd m
.64857 .0873 0 2.5 Mabsadd L
.64857 .0873 1.875 0 Mabsadd L
.64857 .0873 0 -2.5 Mabsadd L
.64857 .0873 -1.875 0 Mabsadd L closepath F
.64952 .0873 -1.875 0 Mabsadd m
.64952 .0873 0 2.5 Mabsadd L
.64952 .0873 1.875 0 Mabsadd L
.64952 .0873 0 -2.5 Mabsadd L
.64952 .0873 -1.875 0 Mabsadd L closepath F
.65048 .08729 -1.875 0 Mabsadd m
.65048 .08729 0 2.5 Mabsadd L
.65048 .08729 1.875 0 Mabsadd L
.65048 .08729 0 -2.5 Mabsadd L
.65048 .08729 -1.875 0 Mabsadd L closepath F
.65143 .08726 -1.875 0 Mabsadd m
.65143 .08726 0 2.5 Mabsadd L
.65143 .08726 1.875 0 Mabsadd L
.65143 .08726 0 -2.5 Mabsadd L
.65143 .08726 -1.875 0 Mabsadd L closepath F
.65238 .08726 -1.875 0 Mabsadd m
.65238 .08726 0 2.5 Mabsadd L
.65238 .08726 1.875 0 Mabsadd L
.65238 .08726 0 -2.5 Mabsadd L
.65238 .08726 -1.875 0 Mabsadd L closepath F
.65333 .0872 -1.875 0 Mabsadd m
.65333 .0872 0 2.5 Mabsadd L
.65333 .0872 1.875 0 Mabsadd L
.65333 .0872 0 -2.5 Mabsadd L
.65333 .0872 -1.875 0 Mabsadd L closepath F
.65429 .08717 -1.875 0 Mabsadd m
.65429 .08717 0 2.5 Mabsadd L
.65429 .08717 1.875 0 Mabsadd L
.65429 .08717 0 -2.5 Mabsadd L
.65429 .08717 -1.875 0 Mabsadd L closepath F
.65524 .08683 -1.875 0 Mabsadd m
.65524 .08683 0 2.5 Mabsadd L
.65524 .08683 1.875 0 Mabsadd L
.65524 .08683 0 -2.5 Mabsadd L
.65524 .08683 -1.875 0 Mabsadd L closepath F
.65619 .0866 -1.875 0 Mabsadd m
.65619 .0866 0 2.5 Mabsadd L
.65619 .0866 1.875 0 Mabsadd L
.65619 .0866 0 -2.5 Mabsadd L
.65619 .0866 -1.875 0 Mabsadd L closepath F
.65714 .08581 -1.875 0 Mabsadd m
.65714 .08581 0 2.5 Mabsadd L
.65714 .08581 1.875 0 Mabsadd L
.65714 .08581 0 -2.5 Mabsadd L
.65714 .08581 -1.875 0 Mabsadd L closepath F
.6581 .08268 -1.875 0 Mabsadd m
.6581 .08268 0 2.5 Mabsadd L
.6581 .08268 1.875 0 Mabsadd L
.6581 .08268 0 -2.5 Mabsadd L
.6581 .08268 -1.875 0 Mabsadd L closepath F
.65905 .06821 -1.875 0 Mabsadd m
.65905 .06821 0 2.5 Mabsadd L
.65905 .06821 1.875 0 Mabsadd L
.65905 .06821 0 -2.5 Mabsadd L
.65905 .06821 -1.875 0 Mabsadd L closepath F
.66 .06819 -1.875 0 Mabsadd m
.66 .06819 0 2.5 Mabsadd L
.66 .06819 1.875 0 Mabsadd L
.66 .06819 0 -2.5 Mabsadd L
.66 .06819 -1.875 0 Mabsadd L closepath F
.66095 .06869 -1.875 0 Mabsadd m
.66095 .06869 0 2.5 Mabsadd L
.66095 .06869 1.875 0 Mabsadd L
.66095 .06869 0 -2.5 Mabsadd L
.66095 .06869 -1.875 0 Mabsadd L closepath F
.6619 .0748 -1.875 0 Mabsadd m
.6619 .0748 0 2.5 Mabsadd L
.6619 .0748 1.875 0 Mabsadd L
.6619 .0748 0 -2.5 Mabsadd L
.6619 .0748 -1.875 0 Mabsadd L closepath F
.66286 .08001 -1.875 0 Mabsadd m
.66286 .08001 0 2.5 Mabsadd L
.66286 .08001 1.875 0 Mabsadd L
.66286 .08001 0 -2.5 Mabsadd L
.66286 .08001 -1.875 0 Mabsadd L closepath F
.66381 .05028 -1.875 0 Mabsadd m
.66381 .05028 0 2.5 Mabsadd L
.66381 .05028 1.875 0 Mabsadd L
.66381 .05028 0 -2.5 Mabsadd L
.66381 .05028 -1.875 0 Mabsadd L closepath F
.66476 .05034 -1.875 0 Mabsadd m
.66476 .05034 0 2.5 Mabsadd L
.66476 .05034 1.875 0 Mabsadd L
.66476 .05034 0 -2.5 Mabsadd L
.66476 .05034 -1.875 0 Mabsadd L closepath F
.66571 .02822 -1.875 0 Mabsadd m
.66571 .02822 0 2.5 Mabsadd L
.66571 .02822 1.875 0 Mabsadd L
.66571 .02822 0 -2.5 Mabsadd L
.66571 .02822 -1.875 0 Mabsadd L closepath F
.66667 .06593 -1.875 0 Mabsadd m
.66667 .06593 0 2.5 Mabsadd L
.66667 .06593 1.875 0 Mabsadd L
.66667 .06593 0 -2.5 Mabsadd L
.66667 .06593 -1.875 0 Mabsadd L closepath F
.66762 .09198 -1.875 0 Mabsadd m
.66762 .09198 0 2.5 Mabsadd L
.66762 .09198 1.875 0 Mabsadd L
.66762 .09198 0 -2.5 Mabsadd L
.66762 .09198 -1.875 0 Mabsadd L closepath F
.66857 .10625 -1.875 0 Mabsadd m
.66857 .10625 0 2.5 Mabsadd L
.66857 .10625 1.875 0 Mabsadd L
.66857 .10625 0 -2.5 Mabsadd L
.66857 .10625 -1.875 0 Mabsadd L closepath F
.66952 .10988 -1.875 0 Mabsadd m
.66952 .10988 0 2.5 Mabsadd L
.66952 .10988 1.875 0 Mabsadd L
.66952 .10988 0 -2.5 Mabsadd L
.66952 .10988 -1.875 0 Mabsadd L closepath F
.67048 .13298 -1.875 0 Mabsadd m
.67048 .13298 0 2.5 Mabsadd L
.67048 .13298 1.875 0 Mabsadd L
.67048 .13298 0 -2.5 Mabsadd L
.67048 .13298 -1.875 0 Mabsadd L closepath F
.67143 .13305 -1.875 0 Mabsadd m
.67143 .13305 0 2.5 Mabsadd L
.67143 .13305 1.875 0 Mabsadd L
.67143 .13305 0 -2.5 Mabsadd L
.67143 .13305 -1.875 0 Mabsadd L closepath F
.67238 .13326 -1.875 0 Mabsadd m
.67238 .13326 0 2.5 Mabsadd L
.67238 .13326 1.875 0 Mabsadd L
.67238 .13326 0 -2.5 Mabsadd L
.67238 .13326 -1.875 0 Mabsadd L closepath F
.67333 .13347 -1.875 0 Mabsadd m
.67333 .13347 0 2.5 Mabsadd L
.67333 .13347 1.875 0 Mabsadd L
.67333 .13347 0 -2.5 Mabsadd L
.67333 .13347 -1.875 0 Mabsadd L closepath F
.67429 .13361 -1.875 0 Mabsadd m
.67429 .13361 0 2.5 Mabsadd L
.67429 .13361 1.875 0 Mabsadd L
.67429 .13361 0 -2.5 Mabsadd L
.67429 .13361 -1.875 0 Mabsadd L closepath F
.67524 .13367 -1.875 0 Mabsadd m
.67524 .13367 0 2.5 Mabsadd L
.67524 .13367 1.875 0 Mabsadd L
.67524 .13367 0 -2.5 Mabsadd L
.67524 .13367 -1.875 0 Mabsadd L closepath F
.67619 .13399 -1.875 0 Mabsadd m
.67619 .13399 0 2.5 Mabsadd L
.67619 .13399 1.875 0 Mabsadd L
.67619 .13399 0 -2.5 Mabsadd L
.67619 .13399 -1.875 0 Mabsadd L closepath F
.67714 .13482 -1.875 0 Mabsadd m
.67714 .13482 0 2.5 Mabsadd L
.67714 .13482 1.875 0 Mabsadd L
.67714 .13482 0 -2.5 Mabsadd L
.67714 .13482 -1.875 0 Mabsadd L closepath F
.6781 .13486 -1.875 0 Mabsadd m
.6781 .13486 0 2.5 Mabsadd L
.6781 .13486 1.875 0 Mabsadd L
.6781 .13486 0 -2.5 Mabsadd L
.6781 .13486 -1.875 0 Mabsadd L closepath F
.67905 .13498 -1.875 0 Mabsadd m
.67905 .13498 0 2.5 Mabsadd L
.67905 .13498 1.875 0 Mabsadd L
.67905 .13498 0 -2.5 Mabsadd L
.67905 .13498 -1.875 0 Mabsadd L closepath F
.68 .13499 -1.875 0 Mabsadd m
.68 .13499 0 2.5 Mabsadd L
.68 .13499 1.875 0 Mabsadd L
.68 .13499 0 -2.5 Mabsadd L
.68 .13499 -1.875 0 Mabsadd L closepath F
.68095 .13502 -1.875 0 Mabsadd m
.68095 .13502 0 2.5 Mabsadd L
.68095 .13502 1.875 0 Mabsadd L
.68095 .13502 0 -2.5 Mabsadd L
.68095 .13502 -1.875 0 Mabsadd L closepath F
.6819 .13502 -1.875 0 Mabsadd m
.6819 .13502 0 2.5 Mabsadd L
.6819 .13502 1.875 0 Mabsadd L
.6819 .13502 0 -2.5 Mabsadd L
.6819 .13502 -1.875 0 Mabsadd L closepath F
.68286 .13503 -1.875 0 Mabsadd m
.68286 .13503 0 2.5 Mabsadd L
.68286 .13503 1.875 0 Mabsadd L
.68286 .13503 0 -2.5 Mabsadd L
.68286 .13503 -1.875 0 Mabsadd L closepath F
.68381 .13504 -1.875 0 Mabsadd m
.68381 .13504 0 2.5 Mabsadd L
.68381 .13504 1.875 0 Mabsadd L
.68381 .13504 0 -2.5 Mabsadd L
.68381 .13504 -1.875 0 Mabsadd L closepath F
.68476 .13504 -1.875 0 Mabsadd m
.68476 .13504 0 2.5 Mabsadd L
.68476 .13504 1.875 0 Mabsadd L
.68476 .13504 0 -2.5 Mabsadd L
.68476 .13504 -1.875 0 Mabsadd L closepath F
.68571 .13504 -1.875 0 Mabsadd m
.68571 .13504 0 2.5 Mabsadd L
.68571 .13504 1.875 0 Mabsadd L
.68571 .13504 0 -2.5 Mabsadd L
.68571 .13504 -1.875 0 Mabsadd L closepath F
.68667 .13505 -1.875 0 Mabsadd m
.68667 .13505 0 2.5 Mabsadd L
.68667 .13505 1.875 0 Mabsadd L
.68667 .13505 0 -2.5 Mabsadd L
.68667 .13505 -1.875 0 Mabsadd L closepath F
.68762 .13505 -1.875 0 Mabsadd m
.68762 .13505 0 2.5 Mabsadd L
.68762 .13505 1.875 0 Mabsadd L
.68762 .13505 0 -2.5 Mabsadd L
.68762 .13505 -1.875 0 Mabsadd L closepath F
.68857 .13505 -1.875 0 Mabsadd m
.68857 .13505 0 2.5 Mabsadd L
.68857 .13505 1.875 0 Mabsadd L
.68857 .13505 0 -2.5 Mabsadd L
.68857 .13505 -1.875 0 Mabsadd L closepath F
.68952 .13505 -1.875 0 Mabsadd m
.68952 .13505 0 2.5 Mabsadd L
.68952 .13505 1.875 0 Mabsadd L
.68952 .13505 0 -2.5 Mabsadd L
.68952 .13505 -1.875 0 Mabsadd L closepath F
.69048 .13505 -1.875 0 Mabsadd m
.69048 .13505 0 2.5 Mabsadd L
.69048 .13505 1.875 0 Mabsadd L
.69048 .13505 0 -2.5 Mabsadd L
.69048 .13505 -1.875 0 Mabsadd L closepath F
.69143 .13505 -1.875 0 Mabsadd m
.69143 .13505 0 2.5 Mabsadd L
.69143 .13505 1.875 0 Mabsadd L
.69143 .13505 0 -2.5 Mabsadd L
.69143 .13505 -1.875 0 Mabsadd L closepath F
.69238 .13505 -1.875 0 Mabsadd m
.69238 .13505 0 2.5 Mabsadd L
.69238 .13505 1.875 0 Mabsadd L
.69238 .13505 0 -2.5 Mabsadd L
.69238 .13505 -1.875 0 Mabsadd L closepath F
.69333 .13505 -1.875 0 Mabsadd m
.69333 .13505 0 2.5 Mabsadd L
.69333 .13505 1.875 0 Mabsadd L
.69333 .13505 0 -2.5 Mabsadd L
.69333 .13505 -1.875 0 Mabsadd L closepath F
.69429 .13505 -1.875 0 Mabsadd m
.69429 .13505 0 2.5 Mabsadd L
.69429 .13505 1.875 0 Mabsadd L
.69429 .13505 0 -2.5 Mabsadd L
.69429 .13505 -1.875 0 Mabsadd L closepath F
.69524 .13505 -1.875 0 Mabsadd m
.69524 .13505 0 2.5 Mabsadd L
.69524 .13505 1.875 0 Mabsadd L
.69524 .13505 0 -2.5 Mabsadd L
.69524 .13505 -1.875 0 Mabsadd L closepath F
.69619 .13505 -1.875 0 Mabsadd m
.69619 .13505 0 2.5 Mabsadd L
.69619 .13505 1.875 0 Mabsadd L
.69619 .13505 0 -2.5 Mabsadd L
.69619 .13505 -1.875 0 Mabsadd L closepath F
.69714 .13505 -1.875 0 Mabsadd m
.69714 .13505 0 2.5 Mabsadd L
.69714 .13505 1.875 0 Mabsadd L
.69714 .13505 0 -2.5 Mabsadd L
.69714 .13505 -1.875 0 Mabsadd L closepath F
.6981 .13505 -1.875 0 Mabsadd m
.6981 .13505 0 2.5 Mabsadd L
.6981 .13505 1.875 0 Mabsadd L
.6981 .13505 0 -2.5 Mabsadd L
.6981 .13505 -1.875 0 Mabsadd L closepath F
.69905 .13504 -1.875 0 Mabsadd m
.69905 .13504 0 2.5 Mabsadd L
.69905 .13504 1.875 0 Mabsadd L
.69905 .13504 0 -2.5 Mabsadd L
.69905 .13504 -1.875 0 Mabsadd L closepath F
.7 .13504 -1.875 0 Mabsadd m
.7 .13504 0 2.5 Mabsadd L
.7 .13504 1.875 0 Mabsadd L
.7 .13504 0 -2.5 Mabsadd L
.7 .13504 -1.875 0 Mabsadd L closepath F
.70095 .13504 -1.875 0 Mabsadd m
.70095 .13504 0 2.5 Mabsadd L
.70095 .13504 1.875 0 Mabsadd L
.70095 .13504 0 -2.5 Mabsadd L
.70095 .13504 -1.875 0 Mabsadd L closepath F
.7019 .13504 -1.875 0 Mabsadd m
.7019 .13504 0 2.5 Mabsadd L
.7019 .13504 1.875 0 Mabsadd L
.7019 .13504 0 -2.5 Mabsadd L
.7019 .13504 -1.875 0 Mabsadd L closepath F
.70286 .13504 -1.875 0 Mabsadd m
.70286 .13504 0 2.5 Mabsadd L
.70286 .13504 1.875 0 Mabsadd L
.70286 .13504 0 -2.5 Mabsadd L
.70286 .13504 -1.875 0 Mabsadd L closepath F
.70381 .13503 -1.875 0 Mabsadd m
.70381 .13503 0 2.5 Mabsadd L
.70381 .13503 1.875 0 Mabsadd L
.70381 .13503 0 -2.5 Mabsadd L
.70381 .13503 -1.875 0 Mabsadd L closepath F
.70476 .135 -1.875 0 Mabsadd m
.70476 .135 0 2.5 Mabsadd L
.70476 .135 1.875 0 Mabsadd L
.70476 .135 0 -2.5 Mabsadd L
.70476 .135 -1.875 0 Mabsadd L closepath F
.70571 .135 -1.875 0 Mabsadd m
.70571 .135 0 2.5 Mabsadd L
.70571 .135 1.875 0 Mabsadd L
.70571 .135 0 -2.5 Mabsadd L
.70571 .135 -1.875 0 Mabsadd L closepath F
.70667 .13494 -1.875 0 Mabsadd m
.70667 .13494 0 2.5 Mabsadd L
.70667 .13494 1.875 0 Mabsadd L
.70667 .13494 0 -2.5 Mabsadd L
.70667 .13494 -1.875 0 Mabsadd L closepath F
.70762 .13492 -1.875 0 Mabsadd m
.70762 .13492 0 2.5 Mabsadd L
.70762 .13492 1.875 0 Mabsadd L
.70762 .13492 0 -2.5 Mabsadd L
.70762 .13492 -1.875 0 Mabsadd L closepath F
.70857 .13462 -1.875 0 Mabsadd m
.70857 .13462 0 2.5 Mabsadd L
.70857 .13462 1.875 0 Mabsadd L
.70857 .13462 0 -2.5 Mabsadd L
.70857 .13462 -1.875 0 Mabsadd L closepath F
.70952 .13442 -1.875 0 Mabsadd m
.70952 .13442 0 2.5 Mabsadd L
.70952 .13442 1.875 0 Mabsadd L
.70952 .13442 0 -2.5 Mabsadd L
.70952 .13442 -1.875 0 Mabsadd L closepath F
.71048 .13376 -1.875 0 Mabsadd m
.71048 .13376 0 2.5 Mabsadd L
.71048 .13376 1.875 0 Mabsadd L
.71048 .13376 0 -2.5 Mabsadd L
.71048 .13376 -1.875 0 Mabsadd L closepath F
.71143 .13116 -1.875 0 Mabsadd m
.71143 .13116 0 2.5 Mabsadd L
.71143 .13116 1.875 0 Mabsadd L
.71143 .13116 0 -2.5 Mabsadd L
.71143 .13116 -1.875 0 Mabsadd L closepath F
.71238 .11939 -1.875 0 Mabsadd m
.71238 .11939 0 2.5 Mabsadd L
.71238 .11939 1.875 0 Mabsadd L
.71238 .11939 0 -2.5 Mabsadd L
.71238 .11939 -1.875 0 Mabsadd L closepath F
.71333 .11938 -1.875 0 Mabsadd m
.71333 .11938 0 2.5 Mabsadd L
.71333 .11938 1.875 0 Mabsadd L
.71333 .11938 0 -2.5 Mabsadd L
.71333 .11938 -1.875 0 Mabsadd L closepath F
.71429 .11976 -1.875 0 Mabsadd m
.71429 .11976 0 2.5 Mabsadd L
.71429 .11976 1.875 0 Mabsadd L
.71429 .11976 0 -2.5 Mabsadd L
.71429 .11976 -1.875 0 Mabsadd L closepath F
.71524 .12466 -1.875 0 Mabsadd m
.71524 .12466 0 2.5 Mabsadd L
.71524 .12466 1.875 0 Mabsadd L
.71524 .12466 0 -2.5 Mabsadd L
.71524 .12466 -1.875 0 Mabsadd L closepath F
.71619 .11569 -1.875 0 Mabsadd m
.71619 .11569 0 2.5 Mabsadd L
.71619 .11569 1.875 0 Mabsadd L
.71619 .11569 0 -2.5 Mabsadd L
.71619 .11569 -1.875 0 Mabsadd L closepath F
.71714 .21483 -1.875 0 Mabsadd m
.71714 .21483 0 2.5 Mabsadd L
.71714 .21483 1.875 0 Mabsadd L
.71714 .21483 0 -2.5 Mabsadd L
.71714 .21483 -1.875 0 Mabsadd L closepath F
.7181 .23301 -1.875 0 Mabsadd m
.7181 .23301 0 2.5 Mabsadd L
.7181 .23301 1.875 0 Mabsadd L
.7181 .23301 0 -2.5 Mabsadd L
.7181 .23301 -1.875 0 Mabsadd L closepath F
.71905 .12506 -1.875 0 Mabsadd m
.71905 .12506 0 2.5 Mabsadd L
.71905 .12506 1.875 0 Mabsadd L
.71905 .12506 0 -2.5 Mabsadd L
.71905 .12506 -1.875 0 Mabsadd L closepath F
.72 .21371 -1.875 0 Mabsadd m
.72 .21371 0 2.5 Mabsadd L
.72 .21371 1.875 0 Mabsadd L
.72 .21371 0 -2.5 Mabsadd L
.72 .21371 -1.875 0 Mabsadd L closepath F
.72095 .23337 -1.875 0 Mabsadd m
.72095 .23337 0 2.5 Mabsadd L
.72095 .23337 1.875 0 Mabsadd L
.72095 .23337 0 -2.5 Mabsadd L
.72095 .23337 -1.875 0 Mabsadd L closepath F
.7219 .23789 -1.875 0 Mabsadd m
.7219 .23789 0 2.5 Mabsadd L
.7219 .23789 1.875 0 Mabsadd L
.7219 .23789 0 -2.5 Mabsadd L
.7219 .23789 -1.875 0 Mabsadd L closepath F
.72286 .24281 -1.875 0 Mabsadd m
.72286 .24281 0 2.5 Mabsadd L
.72286 .24281 1.875 0 Mabsadd L
.72286 .24281 0 -2.5 Mabsadd L
.72286 .24281 -1.875 0 Mabsadd L closepath F
.72381 .24331 -1.875 0 Mabsadd m
.72381 .24331 0 2.5 Mabsadd L
.72381 .24331 1.875 0 Mabsadd L
.72381 .24331 0 -2.5 Mabsadd L
.72381 .24331 -1.875 0 Mabsadd L closepath F
.72476 .24581 -1.875 0 Mabsadd m
.72476 .24581 0 2.5 Mabsadd L
.72476 .24581 1.875 0 Mabsadd L
.72476 .24581 0 -2.5 Mabsadd L
.72476 .24581 -1.875 0 Mabsadd L closepath F
.72571 .24813 -1.875 0 Mabsadd m
.72571 .24813 0 2.5 Mabsadd L
.72571 .24813 1.875 0 Mabsadd L
.72571 .24813 0 -2.5 Mabsadd L
.72571 .24813 -1.875 0 Mabsadd L closepath F
.72667 .24929 -1.875 0 Mabsadd m
.72667 .24929 0 2.5 Mabsadd L
.72667 .24929 1.875 0 Mabsadd L
.72667 .24929 0 -2.5 Mabsadd L
.72667 .24929 -1.875 0 Mabsadd L closepath F
.72762 .254 -1.875 0 Mabsadd m
.72762 .254 0 2.5 Mabsadd L
.72762 .254 1.875 0 Mabsadd L
.72762 .254 0 -2.5 Mabsadd L
.72762 .254 -1.875 0 Mabsadd L closepath F
.72857 .25528 -1.875 0 Mabsadd m
.72857 .25528 0 2.5 Mabsadd L
.72857 .25528 1.875 0 Mabsadd L
.72857 .25528 0 -2.5 Mabsadd L
.72857 .25528 -1.875 0 Mabsadd L closepath F
.72952 .2582 -1.875 0 Mabsadd m
.72952 .2582 0 2.5 Mabsadd L
.72952 .2582 1.875 0 Mabsadd L
.72952 .2582 0 -2.5 Mabsadd L
.72952 .2582 -1.875 0 Mabsadd L closepath F
.73048 .25833 -1.875 0 Mabsadd m
.73048 .25833 0 2.5 Mabsadd L
.73048 .25833 1.875 0 Mabsadd L
.73048 .25833 0 -2.5 Mabsadd L
.73048 .25833 -1.875 0 Mabsadd L closepath F
.73143 .25867 -1.875 0 Mabsadd m
.73143 .25867 0 2.5 Mabsadd L
.73143 .25867 1.875 0 Mabsadd L
.73143 .25867 0 -2.5 Mabsadd L
.73143 .25867 -1.875 0 Mabsadd L closepath F
.73238 .25869 -1.875 0 Mabsadd m
.73238 .25869 0 2.5 Mabsadd L
.73238 .25869 1.875 0 Mabsadd L
.73238 .25869 0 -2.5 Mabsadd L
.73238 .25869 -1.875 0 Mabsadd L closepath F
.73333 .25875 -1.875 0 Mabsadd m
.73333 .25875 0 2.5 Mabsadd L
.73333 .25875 1.875 0 Mabsadd L
.73333 .25875 0 -2.5 Mabsadd L
.73333 .25875 -1.875 0 Mabsadd L closepath F
.73429 .25876 -1.875 0 Mabsadd m
.73429 .25876 0 2.5 Mabsadd L
.73429 .25876 1.875 0 Mabsadd L
.73429 .25876 0 -2.5 Mabsadd L
.73429 .25876 -1.875 0 Mabsadd L closepath F
.73524 .25877 -1.875 0 Mabsadd m
.73524 .25877 0 2.5 Mabsadd L
.73524 .25877 1.875 0 Mabsadd L
.73524 .25877 0 -2.5 Mabsadd L
.73524 .25877 -1.875 0 Mabsadd L closepath F
.73619 .25878 -1.875 0 Mabsadd m
.73619 .25878 0 2.5 Mabsadd L
.73619 .25878 1.875 0 Mabsadd L
.73619 .25878 0 -2.5 Mabsadd L
.73619 .25878 -1.875 0 Mabsadd L closepath F
.73714 .25878 -1.875 0 Mabsadd m
.73714 .25878 0 2.5 Mabsadd L
.73714 .25878 1.875 0 Mabsadd L
.73714 .25878 0 -2.5 Mabsadd L
.73714 .25878 -1.875 0 Mabsadd L closepath F
.7381 .25879 -1.875 0 Mabsadd m
.7381 .25879 0 2.5 Mabsadd L
.7381 .25879 1.875 0 Mabsadd L
.7381 .25879 0 -2.5 Mabsadd L
.7381 .25879 -1.875 0 Mabsadd L closepath F
.73905 .25879 -1.875 0 Mabsadd m
.73905 .25879 0 2.5 Mabsadd L
.73905 .25879 1.875 0 Mabsadd L
.73905 .25879 0 -2.5 Mabsadd L
.73905 .25879 -1.875 0 Mabsadd L closepath F
.74 .25879 -1.875 0 Mabsadd m
.74 .25879 0 2.5 Mabsadd L
.74 .25879 1.875 0 Mabsadd L
.74 .25879 0 -2.5 Mabsadd L
.74 .25879 -1.875 0 Mabsadd L closepath F
.74095 .25879 -1.875 0 Mabsadd m
.74095 .25879 0 2.5 Mabsadd L
.74095 .25879 1.875 0 Mabsadd L
.74095 .25879 0 -2.5 Mabsadd L
.74095 .25879 -1.875 0 Mabsadd L closepath F
.7419 .2588 -1.875 0 Mabsadd m
.7419 .2588 0 2.5 Mabsadd L
.7419 .2588 1.875 0 Mabsadd L
.7419 .2588 0 -2.5 Mabsadd L
.7419 .2588 -1.875 0 Mabsadd L closepath F
.74286 .25879 -1.875 0 Mabsadd m
.74286 .25879 0 2.5 Mabsadd L
.74286 .25879 1.875 0 Mabsadd L
.74286 .25879 0 -2.5 Mabsadd L
.74286 .25879 -1.875 0 Mabsadd L closepath F
.74381 .2588 -1.875 0 Mabsadd m
.74381 .2588 0 2.5 Mabsadd L
.74381 .2588 1.875 0 Mabsadd L
.74381 .2588 0 -2.5 Mabsadd L
.74381 .2588 -1.875 0 Mabsadd L closepath F
.74476 .2588 -1.875 0 Mabsadd m
.74476 .2588 0 2.5 Mabsadd L
.74476 .2588 1.875 0 Mabsadd L
.74476 .2588 0 -2.5 Mabsadd L
.74476 .2588 -1.875 0 Mabsadd L closepath F
.74571 .2588 -1.875 0 Mabsadd m
.74571 .2588 0 2.5 Mabsadd L
.74571 .2588 1.875 0 Mabsadd L
.74571 .2588 0 -2.5 Mabsadd L
.74571 .2588 -1.875 0 Mabsadd L closepath F
.74667 .2588 -1.875 0 Mabsadd m
.74667 .2588 0 2.5 Mabsadd L
.74667 .2588 1.875 0 Mabsadd L
.74667 .2588 0 -2.5 Mabsadd L
.74667 .2588 -1.875 0 Mabsadd L closepath F
.74762 .2588 -1.875 0 Mabsadd m
.74762 .2588 0 2.5 Mabsadd L
.74762 .2588 1.875 0 Mabsadd L
.74762 .2588 0 -2.5 Mabsadd L
.74762 .2588 -1.875 0 Mabsadd L closepath F
.74857 .25879 -1.875 0 Mabsadd m
.74857 .25879 0 2.5 Mabsadd L
.74857 .25879 1.875 0 Mabsadd L
.74857 .25879 0 -2.5 Mabsadd L
.74857 .25879 -1.875 0 Mabsadd L closepath F
.74952 .25879 -1.875 0 Mabsadd m
.74952 .25879 0 2.5 Mabsadd L
.74952 .25879 1.875 0 Mabsadd L
.74952 .25879 0 -2.5 Mabsadd L
.74952 .25879 -1.875 0 Mabsadd L closepath F
.75048 .25879 -1.875 0 Mabsadd m
.75048 .25879 0 2.5 Mabsadd L
.75048 .25879 1.875 0 Mabsadd L
.75048 .25879 0 -2.5 Mabsadd L
.75048 .25879 -1.875 0 Mabsadd L closepath F
.75143 .25879 -1.875 0 Mabsadd m
.75143 .25879 0 2.5 Mabsadd L
.75143 .25879 1.875 0 Mabsadd L
.75143 .25879 0 -2.5 Mabsadd L
.75143 .25879 -1.875 0 Mabsadd L closepath F
.75238 .25879 -1.875 0 Mabsadd m
.75238 .25879 0 2.5 Mabsadd L
.75238 .25879 1.875 0 Mabsadd L
.75238 .25879 0 -2.5 Mabsadd L
.75238 .25879 -1.875 0 Mabsadd L closepath F
.75333 .25879 -1.875 0 Mabsadd m
.75333 .25879 0 2.5 Mabsadd L
.75333 .25879 1.875 0 Mabsadd L
.75333 .25879 0 -2.5 Mabsadd L
.75333 .25879 -1.875 0 Mabsadd L closepath F
.75429 .25879 -1.875 0 Mabsadd m
.75429 .25879 0 2.5 Mabsadd L
.75429 .25879 1.875 0 Mabsadd L
.75429 .25879 0 -2.5 Mabsadd L
.75429 .25879 -1.875 0 Mabsadd L closepath F
.75524 .25878 -1.875 0 Mabsadd m
.75524 .25878 0 2.5 Mabsadd L
.75524 .25878 1.875 0 Mabsadd L
.75524 .25878 0 -2.5 Mabsadd L
.75524 .25878 -1.875 0 Mabsadd L closepath F
.75619 .25877 -1.875 0 Mabsadd m
.75619 .25877 0 2.5 Mabsadd L
.75619 .25877 1.875 0 Mabsadd L
.75619 .25877 0 -2.5 Mabsadd L
.75619 .25877 -1.875 0 Mabsadd L closepath F
.75714 .25877 -1.875 0 Mabsadd m
.75714 .25877 0 2.5 Mabsadd L
.75714 .25877 1.875 0 Mabsadd L
.75714 .25877 0 -2.5 Mabsadd L
.75714 .25877 -1.875 0 Mabsadd L closepath F
.7581 .25876 -1.875 0 Mabsadd m
.7581 .25876 0 2.5 Mabsadd L
.7581 .25876 1.875 0 Mabsadd L
.7581 .25876 0 -2.5 Mabsadd L
.7581 .25876 -1.875 0 Mabsadd L closepath F
.75905 .25876 -1.875 0 Mabsadd m
.75905 .25876 0 2.5 Mabsadd L
.75905 .25876 1.875 0 Mabsadd L
.75905 .25876 0 -2.5 Mabsadd L
.75905 .25876 -1.875 0 Mabsadd L closepath F
.76 .25871 -1.875 0 Mabsadd m
.76 .25871 0 2.5 Mabsadd L
.76 .25871 1.875 0 Mabsadd L
.76 .25871 0 -2.5 Mabsadd L
.76 .25871 -1.875 0 Mabsadd L closepath F
.76095 .25869 -1.875 0 Mabsadd m
.76095 .25869 0 2.5 Mabsadd L
.76095 .25869 1.875 0 Mabsadd L
.76095 .25869 0 -2.5 Mabsadd L
.76095 .25869 -1.875 0 Mabsadd L closepath F
.7619 .25842 -1.875 0 Mabsadd m
.7619 .25842 0 2.5 Mabsadd L
.7619 .25842 1.875 0 Mabsadd L
.7619 .25842 0 -2.5 Mabsadd L
.7619 .25842 -1.875 0 Mabsadd L closepath F
.76286 .25825 -1.875 0 Mabsadd m
.76286 .25825 0 2.5 Mabsadd L
.76286 .25825 1.875 0 Mabsadd L
.76286 .25825 0 -2.5 Mabsadd L
.76286 .25825 -1.875 0 Mabsadd L closepath F
.76381 .2559 -1.875 0 Mabsadd m
.76381 .2559 0 2.5 Mabsadd L
.76381 .2559 1.875 0 Mabsadd L
.76381 .2559 0 -2.5 Mabsadd L
.76381 .2559 -1.875 0 Mabsadd L closepath F
.76476 .25376 -1.875 0 Mabsadd m
.76476 .25376 0 2.5 Mabsadd L
.76476 .25376 1.875 0 Mabsadd L
.76476 .25376 0 -2.5 Mabsadd L
.76476 .25376 -1.875 0 Mabsadd L closepath F
.76571 .24418 -1.875 0 Mabsadd m
.76571 .24418 0 2.5 Mabsadd L
.76571 .24418 1.875 0 Mabsadd L
.76571 .24418 0 -2.5 Mabsadd L
.76571 .24418 -1.875 0 Mabsadd L closepath F
.76667 .19432 -1.875 0 Mabsadd m
.76667 .19432 0 2.5 Mabsadd L
.76667 .19432 1.875 0 Mabsadd L
.76667 .19432 0 -2.5 Mabsadd L
.76667 .19432 -1.875 0 Mabsadd L closepath F
.76762 .19433 -1.875 0 Mabsadd m
.76762 .19433 0 2.5 Mabsadd L
.76762 .19433 1.875 0 Mabsadd L
.76762 .19433 0 -2.5 Mabsadd L
.76762 .19433 -1.875 0 Mabsadd L closepath F
.76857 .19677 -1.875 0 Mabsadd m
.76857 .19677 0 2.5 Mabsadd L
.76857 .19677 1.875 0 Mabsadd L
.76857 .19677 0 -2.5 Mabsadd L
.76857 .19677 -1.875 0 Mabsadd L closepath F
.76952 .21444 -1.875 0 Mabsadd m
.76952 .21444 0 2.5 Mabsadd L
.76952 .21444 1.875 0 Mabsadd L
.76952 .21444 0 -2.5 Mabsadd L
.76952 .21444 -1.875 0 Mabsadd L closepath F
.77048 .2776 -1.875 0 Mabsadd m
.77048 .2776 0 2.5 Mabsadd L
.77048 .2776 1.875 0 Mabsadd L
.77048 .2776 0 -2.5 Mabsadd L
.77048 .2776 -1.875 0 Mabsadd L closepath F
.77143 .29854 -1.875 0 Mabsadd m
.77143 .29854 0 2.5 Mabsadd L
.77143 .29854 1.875 0 Mabsadd L
.77143 .29854 0 -2.5 Mabsadd L
.77143 .29854 -1.875 0 Mabsadd L closepath F
.77238 .13379 -1.875 0 Mabsadd m
.77238 .13379 0 2.5 Mabsadd L
.77238 .13379 1.875 0 Mabsadd L
.77238 .13379 0 -2.5 Mabsadd L
.77238 .13379 -1.875 0 Mabsadd L closepath F
.77333 .27852 -1.875 0 Mabsadd m
.77333 .27852 0 2.5 Mabsadd L
.77333 .27852 1.875 0 Mabsadd L
.77333 .27852 0 -2.5 Mabsadd L
.77333 .27852 -1.875 0 Mabsadd L closepath F
.77429 .28804 -1.875 0 Mabsadd m
.77429 .28804 0 2.5 Mabsadd L
.77429 .28804 1.875 0 Mabsadd L
.77429 .28804 0 -2.5 Mabsadd L
.77429 .28804 -1.875 0 Mabsadd L closepath F
.77524 .27282 -1.875 0 Mabsadd m
.77524 .27282 0 2.5 Mabsadd L
.77524 .27282 1.875 0 Mabsadd L
.77524 .27282 0 -2.5 Mabsadd L
.77524 .27282 -1.875 0 Mabsadd L closepath F
.77619 .27943 -1.875 0 Mabsadd m
.77619 .27943 0 2.5 Mabsadd L
.77619 .27943 1.875 0 Mabsadd L
.77619 .27943 0 -2.5 Mabsadd L
.77619 .27943 -1.875 0 Mabsadd L closepath F
.77714 .28009 -1.875 0 Mabsadd m
.77714 .28009 0 2.5 Mabsadd L
.77714 .28009 1.875 0 Mabsadd L
.77714 .28009 0 -2.5 Mabsadd L
.77714 .28009 -1.875 0 Mabsadd L closepath F
.7781 .28351 -1.875 0 Mabsadd m
.7781 .28351 0 2.5 Mabsadd L
.7781 .28351 1.875 0 Mabsadd L
.7781 .28351 0 -2.5 Mabsadd L
.7781 .28351 -1.875 0 Mabsadd L closepath F
.77905 .28663 -1.875 0 Mabsadd m
.77905 .28663 0 2.5 Mabsadd L
.77905 .28663 1.875 0 Mabsadd L
.77905 .28663 0 -2.5 Mabsadd L
.77905 .28663 -1.875 0 Mabsadd L closepath F
.78 .31185 -1.875 0 Mabsadd m
.78 .31185 0 2.5 Mabsadd L
.78 .31185 1.875 0 Mabsadd L
.78 .31185 0 -2.5 Mabsadd L
.78 .31185 -1.875 0 Mabsadd L closepath F
.78095 .31777 -1.875 0 Mabsadd m
.78095 .31777 0 2.5 Mabsadd L
.78095 .31777 1.875 0 Mabsadd L
.78095 .31777 0 -2.5 Mabsadd L
.78095 .31777 -1.875 0 Mabsadd L closepath F
.7819 .31928 -1.875 0 Mabsadd m
.7819 .31928 0 2.5 Mabsadd L
.7819 .31928 1.875 0 Mabsadd L
.7819 .31928 0 -2.5 Mabsadd L
.7819 .31928 -1.875 0 Mabsadd L closepath F
.78286 .32269 -1.875 0 Mabsadd m
.78286 .32269 0 2.5 Mabsadd L
.78286 .32269 1.875 0 Mabsadd L
.78286 .32269 0 -2.5 Mabsadd L
.78286 .32269 -1.875 0 Mabsadd L closepath F
.78381 .32283 -1.875 0 Mabsadd m
.78381 .32283 0 2.5 Mabsadd L
.78381 .32283 1.875 0 Mabsadd L
.78381 .32283 0 -2.5 Mabsadd L
.78381 .32283 -1.875 0 Mabsadd L closepath F
.78476 .32323 -1.875 0 Mabsadd m
.78476 .32323 0 2.5 Mabsadd L
.78476 .32323 1.875 0 Mabsadd L
.78476 .32323 0 -2.5 Mabsadd L
.78476 .32323 -1.875 0 Mabsadd L closepath F
.78571 .32325 -1.875 0 Mabsadd m
.78571 .32325 0 2.5 Mabsadd L
.78571 .32325 1.875 0 Mabsadd L
.78571 .32325 0 -2.5 Mabsadd L
.78571 .32325 -1.875 0 Mabsadd L closepath F
.78667 .32332 -1.875 0 Mabsadd m
.78667 .32332 0 2.5 Mabsadd L
.78667 .32332 1.875 0 Mabsadd L
.78667 .32332 0 -2.5 Mabsadd L
.78667 .32332 -1.875 0 Mabsadd L closepath F
.78762 .32332 -1.875 0 Mabsadd m
.78762 .32332 0 2.5 Mabsadd L
.78762 .32332 1.875 0 Mabsadd L
.78762 .32332 0 -2.5 Mabsadd L
.78762 .32332 -1.875 0 Mabsadd L closepath F
.78857 .32334 -1.875 0 Mabsadd m
.78857 .32334 0 2.5 Mabsadd L
.78857 .32334 1.875 0 Mabsadd L
.78857 .32334 0 -2.5 Mabsadd L
.78857 .32334 -1.875 0 Mabsadd L closepath F
.78952 .32337 -1.875 0 Mabsadd m
.78952 .32337 0 2.5 Mabsadd L
.78952 .32337 1.875 0 Mabsadd L
.78952 .32337 0 -2.5 Mabsadd L
.78952 .32337 -1.875 0 Mabsadd L closepath F
.79048 .32337 -1.875 0 Mabsadd m
.79048 .32337 0 2.5 Mabsadd L
.79048 .32337 1.875 0 Mabsadd L
.79048 .32337 0 -2.5 Mabsadd L
.79048 .32337 -1.875 0 Mabsadd L closepath F
.79143 .32337 -1.875 0 Mabsadd m
.79143 .32337 0 2.5 Mabsadd L
.79143 .32337 1.875 0 Mabsadd L
.79143 .32337 0 -2.5 Mabsadd L
.79143 .32337 -1.875 0 Mabsadd L closepath F
.79238 .32337 -1.875 0 Mabsadd m
.79238 .32337 0 2.5 Mabsadd L
.79238 .32337 1.875 0 Mabsadd L
.79238 .32337 0 -2.5 Mabsadd L
.79238 .32337 -1.875 0 Mabsadd L closepath F
.79333 .32337 -1.875 0 Mabsadd m
.79333 .32337 0 2.5 Mabsadd L
.79333 .32337 1.875 0 Mabsadd L
.79333 .32337 0 -2.5 Mabsadd L
.79333 .32337 -1.875 0 Mabsadd L closepath F
.79429 .32338 -1.875 0 Mabsadd m
.79429 .32338 0 2.5 Mabsadd L
.79429 .32338 1.875 0 Mabsadd L
.79429 .32338 0 -2.5 Mabsadd L
.79429 .32338 -1.875 0 Mabsadd L closepath F
.79524 .32338 -1.875 0 Mabsadd m
.79524 .32338 0 2.5 Mabsadd L
.79524 .32338 1.875 0 Mabsadd L
.79524 .32338 0 -2.5 Mabsadd L
.79524 .32338 -1.875 0 Mabsadd L closepath F
.79619 .32338 -1.875 0 Mabsadd m
.79619 .32338 0 2.5 Mabsadd L
.79619 .32338 1.875 0 Mabsadd L
.79619 .32338 0 -2.5 Mabsadd L
.79619 .32338 -1.875 0 Mabsadd L closepath F
.79714 .32338 -1.875 0 Mabsadd m
.79714 .32338 0 2.5 Mabsadd L
.79714 .32338 1.875 0 Mabsadd L
.79714 .32338 0 -2.5 Mabsadd L
.79714 .32338 -1.875 0 Mabsadd L closepath F
.7981 .32338 -1.875 0 Mabsadd m
.7981 .32338 0 2.5 Mabsadd L
.7981 .32338 1.875 0 Mabsadd L
.7981 .32338 0 -2.5 Mabsadd L
.7981 .32338 -1.875 0 Mabsadd L closepath F
.79905 .32338 -1.875 0 Mabsadd m
.79905 .32338 0 2.5 Mabsadd L
.79905 .32338 1.875 0 Mabsadd L
.79905 .32338 0 -2.5 Mabsadd L
.79905 .32338 -1.875 0 Mabsadd L closepath F
.8 .32338 -1.875 0 Mabsadd m
.8 .32338 0 2.5 Mabsadd L
.8 .32338 1.875 0 Mabsadd L
.8 .32338 0 -2.5 Mabsadd L
.8 .32338 -1.875 0 Mabsadd L closepath F
.80095 .32338 -1.875 0 Mabsadd m
.80095 .32338 0 2.5 Mabsadd L
.80095 .32338 1.875 0 Mabsadd L
.80095 .32338 0 -2.5 Mabsadd L
.80095 .32338 -1.875 0 Mabsadd L closepath F
.8019 .32338 -1.875 0 Mabsadd m
.8019 .32338 0 2.5 Mabsadd L
.8019 .32338 1.875 0 Mabsadd L
.8019 .32338 0 -2.5 Mabsadd L
.8019 .32338 -1.875 0 Mabsadd L closepath F
.80286 .32338 -1.875 0 Mabsadd m
.80286 .32338 0 2.5 Mabsadd L
.80286 .32338 1.875 0 Mabsadd L
.80286 .32338 0 -2.5 Mabsadd L
.80286 .32338 -1.875 0 Mabsadd L closepath F
.80381 .32338 -1.875 0 Mabsadd m
.80381 .32338 0 2.5 Mabsadd L
.80381 .32338 1.875 0 Mabsadd L
.80381 .32338 0 -2.5 Mabsadd L
.80381 .32338 -1.875 0 Mabsadd L closepath F
.80476 .32338 -1.875 0 Mabsadd m
.80476 .32338 0 2.5 Mabsadd L
.80476 .32338 1.875 0 Mabsadd L
.80476 .32338 0 -2.5 Mabsadd L
.80476 .32338 -1.875 0 Mabsadd L closepath F
.80571 .32337 -1.875 0 Mabsadd m
.80571 .32337 0 2.5 Mabsadd L
.80571 .32337 1.875 0 Mabsadd L
.80571 .32337 0 -2.5 Mabsadd L
.80571 .32337 -1.875 0 Mabsadd L closepath F
.80667 .32337 -1.875 0 Mabsadd m
.80667 .32337 0 2.5 Mabsadd L
.80667 .32337 1.875 0 Mabsadd L
.80667 .32337 0 -2.5 Mabsadd L
.80667 .32337 -1.875 0 Mabsadd L closepath F
.80762 .32337 -1.875 0 Mabsadd m
.80762 .32337 0 2.5 Mabsadd L
.80762 .32337 1.875 0 Mabsadd L
.80762 .32337 0 -2.5 Mabsadd L
.80762 .32337 -1.875 0 Mabsadd L closepath F
.80857 .32337 -1.875 0 Mabsadd m
.80857 .32337 0 2.5 Mabsadd L
.80857 .32337 1.875 0 Mabsadd L
.80857 .32337 0 -2.5 Mabsadd L
.80857 .32337 -1.875 0 Mabsadd L closepath F
.80952 .32336 -1.875 0 Mabsadd m
.80952 .32336 0 2.5 Mabsadd L
.80952 .32336 1.875 0 Mabsadd L
.80952 .32336 0 -2.5 Mabsadd L
.80952 .32336 -1.875 0 Mabsadd L closepath F
.81048 .32336 -1.875 0 Mabsadd m
.81048 .32336 0 2.5 Mabsadd L
.81048 .32336 1.875 0 Mabsadd L
.81048 .32336 0 -2.5 Mabsadd L
.81048 .32336 -1.875 0 Mabsadd L closepath F
.81143 .32335 -1.875 0 Mabsadd m
.81143 .32335 0 2.5 Mabsadd L
.81143 .32335 1.875 0 Mabsadd L
.81143 .32335 0 -2.5 Mabsadd L
.81143 .32335 -1.875 0 Mabsadd L closepath F
.81238 .32334 -1.875 0 Mabsadd m
.81238 .32334 0 2.5 Mabsadd L
.81238 .32334 1.875 0 Mabsadd L
.81238 .32334 0 -2.5 Mabsadd L
.81238 .32334 -1.875 0 Mabsadd L closepath F
.81333 .3233 -1.875 0 Mabsadd m
.81333 .3233 0 2.5 Mabsadd L
.81333 .3233 1.875 0 Mabsadd L
.81333 .3233 0 -2.5 Mabsadd L
.81333 .3233 -1.875 0 Mabsadd L closepath F
.81429 .32328 -1.875 0 Mabsadd m
.81429 .32328 0 2.5 Mabsadd L
.81429 .32328 1.875 0 Mabsadd L
.81429 .32328 0 -2.5 Mabsadd L
.81429 .32328 -1.875 0 Mabsadd L closepath F
.81524 .32305 -1.875 0 Mabsadd m
.81524 .32305 0 2.5 Mabsadd L
.81524 .32305 1.875 0 Mabsadd L
.81524 .32305 0 -2.5 Mabsadd L
.81524 .32305 -1.875 0 Mabsadd L closepath F
.81619 .3229 -1.875 0 Mabsadd m
.81619 .3229 0 2.5 Mabsadd L
.81619 .3229 1.875 0 Mabsadd L
.81619 .3229 0 -2.5 Mabsadd L
.81619 .3229 -1.875 0 Mabsadd L closepath F
.81714 .32089 -1.875 0 Mabsadd m
.81714 .32089 0 2.5 Mabsadd L
.81714 .32089 1.875 0 Mabsadd L
.81714 .32089 0 -2.5 Mabsadd L
.81714 .32089 -1.875 0 Mabsadd L closepath F
.8181 .31911 -1.875 0 Mabsadd m
.8181 .31911 0 2.5 Mabsadd L
.8181 .31911 1.875 0 Mabsadd L
.8181 .31911 0 -2.5 Mabsadd L
.8181 .31911 -1.875 0 Mabsadd L closepath F
.81905 .31127 -1.875 0 Mabsadd m
.81905 .31127 0 2.5 Mabsadd L
.81905 .31127 1.875 0 Mabsadd L
.81905 .31127 0 -2.5 Mabsadd L
.81905 .31127 -1.875 0 Mabsadd L closepath F
.82 .27136 -1.875 0 Mabsadd m
.82 .27136 0 2.5 Mabsadd L
.82 .27136 1.875 0 Mabsadd L
.82 .27136 0 -2.5 Mabsadd L
.82 .27136 -1.875 0 Mabsadd L closepath F
.82095 .27136 -1.875 0 Mabsadd m
.82095 .27136 0 2.5 Mabsadd L
.82095 .27136 1.875 0 Mabsadd L
.82095 .27136 0 -2.5 Mabsadd L
.82095 .27136 -1.875 0 Mabsadd L closepath F
.8219 .27322 -1.875 0 Mabsadd m
.8219 .27322 0 2.5 Mabsadd L
.8219 .27322 1.875 0 Mabsadd L
.8219 .27322 0 -2.5 Mabsadd L
.8219 .27322 -1.875 0 Mabsadd L closepath F
.82286 .28838 -1.875 0 Mabsadd m
.82286 .28838 0 2.5 Mabsadd L
.82286 .28838 1.875 0 Mabsadd L
.82286 .28838 0 -2.5 Mabsadd L
.82286 .28838 -1.875 0 Mabsadd L closepath F
.82381 .32818 -1.875 0 Mabsadd m
.82381 .32818 0 2.5 Mabsadd L
.82381 .32818 1.875 0 Mabsadd L
.82381 .32818 0 -2.5 Mabsadd L
.82381 .32818 -1.875 0 Mabsadd L closepath F
.82476 .34138 -1.875 0 Mabsadd m
.82476 .34138 0 2.5 Mabsadd L
.82476 .34138 1.875 0 Mabsadd L
.82476 .34138 0 -2.5 Mabsadd L
.82476 .34138 -1.875 0 Mabsadd L closepath F
.82571 .15881 -1.875 0 Mabsadd m
.82571 .15881 0 2.5 Mabsadd L
.82571 .15881 1.875 0 Mabsadd L
.82571 .15881 0 -2.5 Mabsadd L
.82571 .15881 -1.875 0 Mabsadd L closepath F
.82667 .34962 -1.875 0 Mabsadd m
.82667 .34962 0 2.5 Mabsadd L
.82667 .34962 1.875 0 Mabsadd L
.82667 .34962 0 -2.5 Mabsadd L
.82667 .34962 -1.875 0 Mabsadd L closepath F
.82762 .34672 -1.875 0 Mabsadd m
.82762 .34672 0 2.5 Mabsadd L
.82762 .34672 1.875 0 Mabsadd L
.82762 .34672 0 -2.5 Mabsadd L
.82762 .34672 -1.875 0 Mabsadd L closepath F
.82857 .36512 -1.875 0 Mabsadd m
.82857 .36512 0 2.5 Mabsadd L
.82857 .36512 1.875 0 Mabsadd L
.82857 .36512 0 -2.5 Mabsadd L
.82857 .36512 -1.875 0 Mabsadd L closepath F
.82952 .3649 -1.875 0 Mabsadd m
.82952 .3649 0 2.5 Mabsadd L
.82952 .3649 1.875 0 Mabsadd L
.82952 .3649 0 -2.5 Mabsadd L
.82952 .3649 -1.875 0 Mabsadd L closepath F
.83048 .36578 -1.875 0 Mabsadd m
.83048 .36578 0 2.5 Mabsadd L
.83048 .36578 1.875 0 Mabsadd L
.83048 .36578 0 -2.5 Mabsadd L
.83048 .36578 -1.875 0 Mabsadd L closepath F
.83143 .37048 -1.875 0 Mabsadd m
.83143 .37048 0 2.5 Mabsadd L
.83143 .37048 1.875 0 Mabsadd L
.83143 .37048 0 -2.5 Mabsadd L
.83143 .37048 -1.875 0 Mabsadd L closepath F
.83238 .37469 -1.875 0 Mabsadd m
.83238 .37469 0 2.5 Mabsadd L
.83238 .37469 1.875 0 Mabsadd L
.83238 .37469 0 -2.5 Mabsadd L
.83238 .37469 -1.875 0 Mabsadd L closepath F
.83333 .40559 -1.875 0 Mabsadd m
.83333 .40559 0 2.5 Mabsadd L
.83333 .40559 1.875 0 Mabsadd L
.83333 .40559 0 -2.5 Mabsadd L
.83333 .40559 -1.875 0 Mabsadd L closepath F
.83429 .41274 -1.875 0 Mabsadd m
.83429 .41274 0 2.5 Mabsadd L
.83429 .41274 1.875 0 Mabsadd L
.83429 .41274 0 -2.5 Mabsadd L
.83429 .41274 -1.875 0 Mabsadd L closepath F
.83524 .41452 -1.875 0 Mabsadd m
.83524 .41452 0 2.5 Mabsadd L
.83524 .41452 1.875 0 Mabsadd L
.83524 .41452 0 -2.5 Mabsadd L
.83524 .41452 -1.875 0 Mabsadd L closepath F
.83619 .41503 -1.875 0 Mabsadd m
.83619 .41503 0 2.5 Mabsadd L
.83619 .41503 1.875 0 Mabsadd L
.83619 .41503 0 -2.5 Mabsadd L
.83619 .41503 -1.875 0 Mabsadd L closepath F
.83714 .41628 -1.875 0 Mabsadd m
.83714 .41628 0 2.5 Mabsadd L
.83714 .41628 1.875 0 Mabsadd L
.83714 .41628 0 -2.5 Mabsadd L
.83714 .41628 -1.875 0 Mabsadd L closepath F
.8381 .41635 -1.875 0 Mabsadd m
.8381 .41635 0 2.5 Mabsadd L
.8381 .41635 1.875 0 Mabsadd L
.8381 .41635 0 -2.5 Mabsadd L
.8381 .41635 -1.875 0 Mabsadd L closepath F
.83905 .41652 -1.875 0 Mabsadd m
.83905 .41652 0 2.5 Mabsadd L
.83905 .41652 1.875 0 Mabsadd L
.83905 .41652 0 -2.5 Mabsadd L
.83905 .41652 -1.875 0 Mabsadd L closepath F
.84 .41653 -1.875 0 Mabsadd m
.84 .41653 0 2.5 Mabsadd L
.84 .41653 1.875 0 Mabsadd L
.84 .41653 0 -2.5 Mabsadd L
.84 .41653 -1.875 0 Mabsadd L closepath F
.84095 .41657 -1.875 0 Mabsadd m
.84095 .41657 0 2.5 Mabsadd L
.84095 .41657 1.875 0 Mabsadd L
.84095 .41657 0 -2.5 Mabsadd L
.84095 .41657 -1.875 0 Mabsadd L closepath F
.8419 .41657 -1.875 0 Mabsadd m
.8419 .41657 0 2.5 Mabsadd L
.8419 .41657 1.875 0 Mabsadd L
.8419 .41657 0 -2.5 Mabsadd L
.8419 .41657 -1.875 0 Mabsadd L closepath F
.84286 .41658 -1.875 0 Mabsadd m
.84286 .41658 0 2.5 Mabsadd L
.84286 .41658 1.875 0 Mabsadd L
.84286 .41658 0 -2.5 Mabsadd L
.84286 .41658 -1.875 0 Mabsadd L closepath F
.84381 .4166 -1.875 0 Mabsadd m
.84381 .4166 0 2.5 Mabsadd L
.84381 .4166 1.875 0 Mabsadd L
.84381 .4166 0 -2.5 Mabsadd L
.84381 .4166 -1.875 0 Mabsadd L closepath F
.84476 .4166 -1.875 0 Mabsadd m
.84476 .4166 0 2.5 Mabsadd L
.84476 .4166 1.875 0 Mabsadd L
.84476 .4166 0 -2.5 Mabsadd L
.84476 .4166 -1.875 0 Mabsadd L closepath F
.84571 .4166 -1.875 0 Mabsadd m
.84571 .4166 0 2.5 Mabsadd L
.84571 .4166 1.875 0 Mabsadd L
.84571 .4166 0 -2.5 Mabsadd L
.84571 .4166 -1.875 0 Mabsadd L closepath F
.84667 .4166 -1.875 0 Mabsadd m
.84667 .4166 0 2.5 Mabsadd L
.84667 .4166 1.875 0 Mabsadd L
.84667 .4166 0 -2.5 Mabsadd L
.84667 .4166 -1.875 0 Mabsadd L closepath F
.84762 .4166 -1.875 0 Mabsadd m
.84762 .4166 0 2.5 Mabsadd L
.84762 .4166 1.875 0 Mabsadd L
.84762 .4166 0 -2.5 Mabsadd L
.84762 .4166 -1.875 0 Mabsadd L closepath F
.84857 .4166 -1.875 0 Mabsadd m
.84857 .4166 0 2.5 Mabsadd L
.84857 .4166 1.875 0 Mabsadd L
.84857 .4166 0 -2.5 Mabsadd L
.84857 .4166 -1.875 0 Mabsadd L closepath F
.84952 .4166 -1.875 0 Mabsadd m
.84952 .4166 0 2.5 Mabsadd L
.84952 .4166 1.875 0 Mabsadd L
.84952 .4166 0 -2.5 Mabsadd L
.84952 .4166 -1.875 0 Mabsadd L closepath F
.85048 .41661 -1.875 0 Mabsadd m
.85048 .41661 0 2.5 Mabsadd L
.85048 .41661 1.875 0 Mabsadd L
.85048 .41661 0 -2.5 Mabsadd L
.85048 .41661 -1.875 0 Mabsadd L closepath F
.85143 .41661 -1.875 0 Mabsadd m
.85143 .41661 0 2.5 Mabsadd L
.85143 .41661 1.875 0 Mabsadd L
.85143 .41661 0 -2.5 Mabsadd L
.85143 .41661 -1.875 0 Mabsadd L closepath F
.85238 .41661 -1.875 0 Mabsadd m
.85238 .41661 0 2.5 Mabsadd L
.85238 .41661 1.875 0 Mabsadd L
.85238 .41661 0 -2.5 Mabsadd L
.85238 .41661 -1.875 0 Mabsadd L closepath F
.85333 .41661 -1.875 0 Mabsadd m
.85333 .41661 0 2.5 Mabsadd L
.85333 .41661 1.875 0 Mabsadd L
.85333 .41661 0 -2.5 Mabsadd L
.85333 .41661 -1.875 0 Mabsadd L closepath F
.85429 .41661 -1.875 0 Mabsadd m
.85429 .41661 0 2.5 Mabsadd L
.85429 .41661 1.875 0 Mabsadd L
.85429 .41661 0 -2.5 Mabsadd L
.85429 .41661 -1.875 0 Mabsadd L closepath F
.85524 .41661 -1.875 0 Mabsadd m
.85524 .41661 0 2.5 Mabsadd L
.85524 .41661 1.875 0 Mabsadd L
.85524 .41661 0 -2.5 Mabsadd L
.85524 .41661 -1.875 0 Mabsadd L closepath F
.85619 .4166 -1.875 0 Mabsadd m
.85619 .4166 0 2.5 Mabsadd L
.85619 .4166 1.875 0 Mabsadd L
.85619 .4166 0 -2.5 Mabsadd L
.85619 .4166 -1.875 0 Mabsadd L closepath F
.85714 .4166 -1.875 0 Mabsadd m
.85714 .4166 0 2.5 Mabsadd L
.85714 .4166 1.875 0 Mabsadd L
.85714 .4166 0 -2.5 Mabsadd L
.85714 .4166 -1.875 0 Mabsadd L closepath F
.8581 .4166 -1.875 0 Mabsadd m
.8581 .4166 0 2.5 Mabsadd L
.8581 .4166 1.875 0 Mabsadd L
.8581 .4166 0 -2.5 Mabsadd L
.8581 .4166 -1.875 0 Mabsadd L closepath F
.85905 .4166 -1.875 0 Mabsadd m
.85905 .4166 0 2.5 Mabsadd L
.85905 .4166 1.875 0 Mabsadd L
.85905 .4166 0 -2.5 Mabsadd L
.85905 .4166 -1.875 0 Mabsadd L closepath F
.86 .4166 -1.875 0 Mabsadd m
.86 .4166 0 2.5 Mabsadd L
.86 .4166 1.875 0 Mabsadd L
.86 .4166 0 -2.5 Mabsadd L
.86 .4166 -1.875 0 Mabsadd L closepath F
.86095 .4166 -1.875 0 Mabsadd m
.86095 .4166 0 2.5 Mabsadd L
.86095 .4166 1.875 0 Mabsadd L
.86095 .4166 0 -2.5 Mabsadd L
.86095 .4166 -1.875 0 Mabsadd L closepath F
.8619 .41659 -1.875 0 Mabsadd m
.8619 .41659 0 2.5 Mabsadd L
.8619 .41659 1.875 0 Mabsadd L
.8619 .41659 0 -2.5 Mabsadd L
.8619 .41659 -1.875 0 Mabsadd L closepath F
.86286 .41659 -1.875 0 Mabsadd m
.86286 .41659 0 2.5 Mabsadd L
.86286 .41659 1.875 0 Mabsadd L
.86286 .41659 0 -2.5 Mabsadd L
.86286 .41659 -1.875 0 Mabsadd L closepath F
.86381 .41658 -1.875 0 Mabsadd m
.86381 .41658 0 2.5 Mabsadd L
.86381 .41658 1.875 0 Mabsadd L
.86381 .41658 0 -2.5 Mabsadd L
.86381 .41658 -1.875 0 Mabsadd L closepath F
.86476 .41658 -1.875 0 Mabsadd m
.86476 .41658 0 2.5 Mabsadd L
.86476 .41658 1.875 0 Mabsadd L
.86476 .41658 0 -2.5 Mabsadd L
.86476 .41658 -1.875 0 Mabsadd L closepath F
.86571 .41657 -1.875 0 Mabsadd m
.86571 .41657 0 2.5 Mabsadd L
.86571 .41657 1.875 0 Mabsadd L
.86571 .41657 0 -2.5 Mabsadd L
.86571 .41657 -1.875 0 Mabsadd L closepath F
.86667 .41653 -1.875 0 Mabsadd m
.86667 .41653 0 2.5 Mabsadd L
.86667 .41653 1.875 0 Mabsadd L
.86667 .41653 0 -2.5 Mabsadd L
.86667 .41653 -1.875 0 Mabsadd L closepath F
.86762 .41651 -1.875 0 Mabsadd m
.86762 .41651 0 2.5 Mabsadd L
.86762 .41651 1.875 0 Mabsadd L
.86762 .41651 0 -2.5 Mabsadd L
.86762 .41651 -1.875 0 Mabsadd L closepath F
.86857 .41631 -1.875 0 Mabsadd m
.86857 .41631 0 2.5 Mabsadd L
.86857 .41631 1.875 0 Mabsadd L
.86857 .41631 0 -2.5 Mabsadd L
.86857 .41631 -1.875 0 Mabsadd L closepath F
.86952 .41618 -1.875 0 Mabsadd m
.86952 .41618 0 2.5 Mabsadd L
.86952 .41618 1.875 0 Mabsadd L
.86952 .41618 0 -2.5 Mabsadd L
.86952 .41618 -1.875 0 Mabsadd L closepath F
.87048 .41446 -1.875 0 Mabsadd m
.87048 .41446 0 2.5 Mabsadd L
.87048 .41446 1.875 0 Mabsadd L
.87048 .41446 0 -2.5 Mabsadd L
.87048 .41446 -1.875 0 Mabsadd L closepath F
.87143 .41297 -1.875 0 Mabsadd m
.87143 .41297 0 2.5 Mabsadd L
.87143 .41297 1.875 0 Mabsadd L
.87143 .41297 0 -2.5 Mabsadd L
.87143 .41297 -1.875 0 Mabsadd L closepath F
.87238 .38766 -1.875 0 Mabsadd m
.87238 .38766 0 2.5 Mabsadd L
.87238 .38766 1.875 0 Mabsadd L
.87238 .38766 0 -2.5 Mabsadd L
.87238 .38766 -1.875 0 Mabsadd L closepath F
.87333 .35604 -1.875 0 Mabsadd m
.87333 .35604 0 2.5 Mabsadd L
.87333 .35604 1.875 0 Mabsadd L
.87333 .35604 0 -2.5 Mabsadd L
.87333 .35604 -1.875 0 Mabsadd L closepath F
.87429 .35603 -1.875 0 Mabsadd m
.87429 .35603 0 2.5 Mabsadd L
.87429 .35603 1.875 0 Mabsadd L
.87429 .35603 0 -2.5 Mabsadd L
.87429 .35603 -1.875 0 Mabsadd L closepath F
.87524 .35744 -1.875 0 Mabsadd m
.87524 .35744 0 2.5 Mabsadd L
.87524 .35744 1.875 0 Mabsadd L
.87524 .35744 0 -2.5 Mabsadd L
.87524 .35744 -1.875 0 Mabsadd L closepath F
.87619 .37023 -1.875 0 Mabsadd m
.87619 .37023 0 2.5 Mabsadd L
.87619 .37023 1.875 0 Mabsadd L
.87619 .37023 0 -2.5 Mabsadd L
.87619 .37023 -1.875 0 Mabsadd L closepath F
.87714 .39523 -1.875 0 Mabsadd m
.87714 .39523 0 2.5 Mabsadd L
.87714 .39523 1.875 0 Mabsadd L
.87714 .39523 0 -2.5 Mabsadd L
.87714 .39523 -1.875 0 Mabsadd L closepath F
.8781 .39591 -1.875 0 Mabsadd m
.8781 .39591 0 2.5 Mabsadd L
.8781 .39591 1.875 0 Mabsadd L
.8781 .39591 0 -2.5 Mabsadd L
.8781 .39591 -1.875 0 Mabsadd L closepath F
.87905 .23211 -1.875 0 Mabsadd m
.87905 .23211 0 2.5 Mabsadd L
.87905 .23211 1.875 0 Mabsadd L
.87905 .23211 0 -2.5 Mabsadd L
.87905 .23211 -1.875 0 Mabsadd L closepath F
.88 .44252 -1.875 0 Mabsadd m
.88 .44252 0 2.5 Mabsadd L
.88 .44252 1.875 0 Mabsadd L
.88 .44252 0 -2.5 Mabsadd L
.88 .44252 -1.875 0 Mabsadd L closepath F
.88095 .4282 -1.875 0 Mabsadd m
.88095 .4282 0 2.5 Mabsadd L
.88095 .4282 1.875 0 Mabsadd L
.88095 .4282 0 -2.5 Mabsadd L
.88095 .4282 -1.875 0 Mabsadd L closepath F
.8819 .4507 -1.875 0 Mabsadd m
.8819 .4507 0 2.5 Mabsadd L
.8819 .4507 1.875 0 Mabsadd L
.8819 .4507 0 -2.5 Mabsadd L
.8819 .4507 -1.875 0 Mabsadd L closepath F
.88286 .4502 -1.875 0 Mabsadd m
.88286 .4502 0 2.5 Mabsadd L
.88286 .4502 1.875 0 Mabsadd L
.88286 .4502 0 -2.5 Mabsadd L
.88286 .4502 -1.875 0 Mabsadd L closepath F
.88381 .45136 -1.875 0 Mabsadd m
.88381 .45136 0 2.5 Mabsadd L
.88381 .45136 1.875 0 Mabsadd L
.88381 .45136 0 -2.5 Mabsadd L
.88381 .45136 -1.875 0 Mabsadd L closepath F
.88476 .45788 -1.875 0 Mabsadd m
.88476 .45788 0 2.5 Mabsadd L
.88476 .45788 1.875 0 Mabsadd L
.88476 .45788 0 -2.5 Mabsadd L
.88476 .45788 -1.875 0 Mabsadd L closepath F
.88571 .45792 -1.875 0 Mabsadd m
.88571 .45792 0 2.5 Mabsadd L
.88571 .45792 1.875 0 Mabsadd L
.88571 .45792 0 -2.5 Mabsadd L
.88571 .45792 -1.875 0 Mabsadd L closepath F
.88667 .458 -1.875 0 Mabsadd m
.88667 .458 0 2.5 Mabsadd L
.88667 .458 1.875 0 Mabsadd L
.88667 .458 0 -2.5 Mabsadd L
.88667 .458 -1.875 0 Mabsadd L closepath F
.88762 .45809 -1.875 0 Mabsadd m
.88762 .45809 0 2.5 Mabsadd L
.88762 .45809 1.875 0 Mabsadd L
.88762 .45809 0 -2.5 Mabsadd L
.88762 .45809 -1.875 0 Mabsadd L closepath F
.88857 .45815 -1.875 0 Mabsadd m
.88857 .45815 0 2.5 Mabsadd L
.88857 .45815 1.875 0 Mabsadd L
.88857 .45815 0 -2.5 Mabsadd L
.88857 .45815 -1.875 0 Mabsadd L closepath F
.88952 .45818 -1.875 0 Mabsadd m
.88952 .45818 0 2.5 Mabsadd L
.88952 .45818 1.875 0 Mabsadd L
.88952 .45818 0 -2.5 Mabsadd L
.88952 .45818 -1.875 0 Mabsadd L closepath F
.89048 .45818 -1.875 0 Mabsadd m
.89048 .45818 0 2.5 Mabsadd L
.89048 .45818 1.875 0 Mabsadd L
.89048 .45818 0 -2.5 Mabsadd L
.89048 .45818 -1.875 0 Mabsadd L closepath F
.89143 .45825 -1.875 0 Mabsadd m
.89143 .45825 0 2.5 Mabsadd L
.89143 .45825 1.875 0 Mabsadd L
.89143 .45825 0 -2.5 Mabsadd L
.89143 .45825 -1.875 0 Mabsadd L closepath F
.89238 .45845 -1.875 0 Mabsadd m
.89238 .45845 0 2.5 Mabsadd L
.89238 .45845 1.875 0 Mabsadd L
.89238 .45845 0 -2.5 Mabsadd L
.89238 .45845 -1.875 0 Mabsadd L closepath F
.89333 .45846 -1.875 0 Mabsadd m
.89333 .45846 0 2.5 Mabsadd L
.89333 .45846 1.875 0 Mabsadd L
.89333 .45846 0 -2.5 Mabsadd L
.89333 .45846 -1.875 0 Mabsadd L closepath F
.89429 .4585 -1.875 0 Mabsadd m
.89429 .4585 0 2.5 Mabsadd L
.89429 .4585 1.875 0 Mabsadd L
.89429 .4585 0 -2.5 Mabsadd L
.89429 .4585 -1.875 0 Mabsadd L closepath F
.89524 .4585 -1.875 0 Mabsadd m
.89524 .4585 0 2.5 Mabsadd L
.89524 .4585 1.875 0 Mabsadd L
.89524 .4585 0 -2.5 Mabsadd L
.89524 .4585 -1.875 0 Mabsadd L closepath F
.89619 .45851 -1.875 0 Mabsadd m
.89619 .45851 0 2.5 Mabsadd L
.89619 .45851 1.875 0 Mabsadd L
.89619 .45851 0 -2.5 Mabsadd L
.89619 .45851 -1.875 0 Mabsadd L closepath F
.89714 .45853 -1.875 0 Mabsadd m
.89714 .45853 0 2.5 Mabsadd L
.89714 .45853 1.875 0 Mabsadd L
.89714 .45853 0 -2.5 Mabsadd L
.89714 .45853 -1.875 0 Mabsadd L closepath F
.8981 .45853 -1.875 0 Mabsadd m
.8981 .45853 0 2.5 Mabsadd L
.8981 .45853 1.875 0 Mabsadd L
.8981 .45853 0 -2.5 Mabsadd L
.8981 .45853 -1.875 0 Mabsadd L closepath F
.89905 .45853 -1.875 0 Mabsadd m
.89905 .45853 0 2.5 Mabsadd L
.89905 .45853 1.875 0 Mabsadd L
.89905 .45853 0 -2.5 Mabsadd L
.89905 .45853 -1.875 0 Mabsadd L closepath F
.9 .45854 -1.875 0 Mabsadd m
.9 .45854 0 2.5 Mabsadd L
.9 .45854 1.875 0 Mabsadd L
.9 .45854 0 -2.5 Mabsadd L
.9 .45854 -1.875 0 Mabsadd L closepath F
.90095 .45854 -1.875 0 Mabsadd m
.90095 .45854 0 2.5 Mabsadd L
.90095 .45854 1.875 0 Mabsadd L
.90095 .45854 0 -2.5 Mabsadd L
.90095 .45854 -1.875 0 Mabsadd L closepath F
.9019 .45854 -1.875 0 Mabsadd m
.9019 .45854 0 2.5 Mabsadd L
.9019 .45854 1.875 0 Mabsadd L
.9019 .45854 0 -2.5 Mabsadd L
.9019 .45854 -1.875 0 Mabsadd L closepath F
.90286 .45854 -1.875 0 Mabsadd m
.90286 .45854 0 2.5 Mabsadd L
.90286 .45854 1.875 0 Mabsadd L
.90286 .45854 0 -2.5 Mabsadd L
.90286 .45854 -1.875 0 Mabsadd L closepath F
.90381 .45854 -1.875 0 Mabsadd m
.90381 .45854 0 2.5 Mabsadd L
.90381 .45854 1.875 0 Mabsadd L
.90381 .45854 0 -2.5 Mabsadd L
.90381 .45854 -1.875 0 Mabsadd L closepath F
.90476 .45854 -1.875 0 Mabsadd m
.90476 .45854 0 2.5 Mabsadd L
.90476 .45854 1.875 0 Mabsadd L
.90476 .45854 0 -2.5 Mabsadd L
.90476 .45854 -1.875 0 Mabsadd L closepath F
.90571 .45854 -1.875 0 Mabsadd m
.90571 .45854 0 2.5 Mabsadd L
.90571 .45854 1.875 0 Mabsadd L
.90571 .45854 0 -2.5 Mabsadd L
.90571 .45854 -1.875 0 Mabsadd L closepath F
.90667 .45854 -1.875 0 Mabsadd m
.90667 .45854 0 2.5 Mabsadd L
.90667 .45854 1.875 0 Mabsadd L
.90667 .45854 0 -2.5 Mabsadd L
.90667 .45854 -1.875 0 Mabsadd L closepath F
.90762 .45854 -1.875 0 Mabsadd m
.90762 .45854 0 2.5 Mabsadd L
.90762 .45854 1.875 0 Mabsadd L
.90762 .45854 0 -2.5 Mabsadd L
.90762 .45854 -1.875 0 Mabsadd L closepath F
.90857 .45854 -1.875 0 Mabsadd m
.90857 .45854 0 2.5 Mabsadd L
.90857 .45854 1.875 0 Mabsadd L
.90857 .45854 0 -2.5 Mabsadd L
.90857 .45854 -1.875 0 Mabsadd L closepath F
.90952 .45854 -1.875 0 Mabsadd m
.90952 .45854 0 2.5 Mabsadd L
.90952 .45854 1.875 0 Mabsadd L
.90952 .45854 0 -2.5 Mabsadd L
.90952 .45854 -1.875 0 Mabsadd L closepath F
.91048 .45854 -1.875 0 Mabsadd m
.91048 .45854 0 2.5 Mabsadd L
.91048 .45854 1.875 0 Mabsadd L
.91048 .45854 0 -2.5 Mabsadd L
.91048 .45854 -1.875 0 Mabsadd L closepath F
.91143 .45854 -1.875 0 Mabsadd m
.91143 .45854 0 2.5 Mabsadd L
.91143 .45854 1.875 0 Mabsadd L
.91143 .45854 0 -2.5 Mabsadd L
.91143 .45854 -1.875 0 Mabsadd L closepath F
.91238 .45854 -1.875 0 Mabsadd m
.91238 .45854 0 2.5 Mabsadd L
.91238 .45854 1.875 0 Mabsadd L
.91238 .45854 0 -2.5 Mabsadd L
.91238 .45854 -1.875 0 Mabsadd L closepath F
.91333 .45854 -1.875 0 Mabsadd m
.91333 .45854 0 2.5 Mabsadd L
.91333 .45854 1.875 0 Mabsadd L
.91333 .45854 0 -2.5 Mabsadd L
.91333 .45854 -1.875 0 Mabsadd L closepath F
.91429 .45853 -1.875 0 Mabsadd m
.91429 .45853 0 2.5 Mabsadd L
.91429 .45853 1.875 0 Mabsadd L
.91429 .45853 0 -2.5 Mabsadd L
.91429 .45853 -1.875 0 Mabsadd L closepath F
.91524 .45853 -1.875 0 Mabsadd m
.91524 .45853 0 2.5 Mabsadd L
.91524 .45853 1.875 0 Mabsadd L
.91524 .45853 0 -2.5 Mabsadd L
.91524 .45853 -1.875 0 Mabsadd L closepath F
.91619 .45853 -1.875 0 Mabsadd m
.91619 .45853 0 2.5 Mabsadd L
.91619 .45853 1.875 0 Mabsadd L
.91619 .45853 0 -2.5 Mabsadd L
.91619 .45853 -1.875 0 Mabsadd L closepath F
.91714 .45852 -1.875 0 Mabsadd m
.91714 .45852 0 2.5 Mabsadd L
.91714 .45852 1.875 0 Mabsadd L
.91714 .45852 0 -2.5 Mabsadd L
.91714 .45852 -1.875 0 Mabsadd L closepath F
.9181 .45851 -1.875 0 Mabsadd m
.9181 .45851 0 2.5 Mabsadd L
.9181 .45851 1.875 0 Mabsadd L
.9181 .45851 0 -2.5 Mabsadd L
.9181 .45851 -1.875 0 Mabsadd L closepath F
.91905 .4585 -1.875 0 Mabsadd m
.91905 .4585 0 2.5 Mabsadd L
.91905 .4585 1.875 0 Mabsadd L
.91905 .4585 0 -2.5 Mabsadd L
.91905 .4585 -1.875 0 Mabsadd L closepath F
.92 .45842 -1.875 0 Mabsadd m
.92 .45842 0 2.5 Mabsadd L
.92 .45842 1.875 0 Mabsadd L
.92 .45842 0 -2.5 Mabsadd L
.92 .45842 -1.875 0 Mabsadd L closepath F
.92095 .45841 -1.875 0 Mabsadd m
.92095 .45841 0 2.5 Mabsadd L
.92095 .45841 1.875 0 Mabsadd L
.92095 .45841 0 -2.5 Mabsadd L
.92095 .45841 -1.875 0 Mabsadd L closepath F
.9219 .45823 -1.875 0 Mabsadd m
.9219 .45823 0 2.5 Mabsadd L
.9219 .45823 1.875 0 Mabsadd L
.9219 .45823 0 -2.5 Mabsadd L
.9219 .45823 -1.875 0 Mabsadd L closepath F
.92286 .45811 -1.875 0 Mabsadd m
.92286 .45811 0 2.5 Mabsadd L
.92286 .45811 1.875 0 Mabsadd L
.92286 .45811 0 -2.5 Mabsadd L
.92286 .45811 -1.875 0 Mabsadd L closepath F
.92381 .45664 -1.875 0 Mabsadd m
.92381 .45664 0 2.5 Mabsadd L
.92381 .45664 1.875 0 Mabsadd L
.92381 .45664 0 -2.5 Mabsadd L
.92381 .45664 -1.875 0 Mabsadd L closepath F
.92476 .45539 -1.875 0 Mabsadd m
.92476 .45539 0 2.5 Mabsadd L
.92476 .45539 1.875 0 Mabsadd L
.92476 .45539 0 -2.5 Mabsadd L
.92476 .45539 -1.875 0 Mabsadd L closepath F
.92571 .43451 -1.875 0 Mabsadd m
.92571 .43451 0 2.5 Mabsadd L
.92571 .43451 1.875 0 Mabsadd L
.92571 .43451 0 -2.5 Mabsadd L
.92571 .43451 -1.875 0 Mabsadd L closepath F
.92667 .40906 -1.875 0 Mabsadd m
.92667 .40906 0 2.5 Mabsadd L
.92667 .40906 1.875 0 Mabsadd L
.92667 .40906 0 -2.5 Mabsadd L
.92667 .40906 -1.875 0 Mabsadd L closepath F
.92762 .40904 -1.875 0 Mabsadd m
.92762 .40904 0 2.5 Mabsadd L
.92762 .40904 1.875 0 Mabsadd L
.92762 .40904 0 -2.5 Mabsadd L
.92762 .40904 -1.875 0 Mabsadd L closepath F
.92857 .41012 -1.875 0 Mabsadd m
.92857 .41012 0 2.5 Mabsadd L
.92857 .41012 1.875 0 Mabsadd L
.92857 .41012 0 -2.5 Mabsadd L
.92857 .41012 -1.875 0 Mabsadd L closepath F
.92952 .42075 -1.875 0 Mabsadd m
.92952 .42075 0 2.5 Mabsadd L
.92952 .42075 1.875 0 Mabsadd L
.92952 .42075 0 -2.5 Mabsadd L
.92952 .42075 -1.875 0 Mabsadd L closepath F
.93048 .43658 -1.875 0 Mabsadd m
.93048 .43658 0 2.5 Mabsadd L
.93048 .43658 1.875 0 Mabsadd L
.93048 .43658 0 -2.5 Mabsadd L
.93048 .43658 -1.875 0 Mabsadd L closepath F
.93143 .42463 -1.875 0 Mabsadd m
.93143 .42463 0 2.5 Mabsadd L
.93143 .42463 1.875 0 Mabsadd L
.93143 .42463 0 -2.5 Mabsadd L
.93143 .42463 -1.875 0 Mabsadd L closepath F
.93238 .30405 -1.875 0 Mabsadd m
.93238 .30405 0 2.5 Mabsadd L
.93238 .30405 1.875 0 Mabsadd L
.93238 .30405 0 -2.5 Mabsadd L
.93238 .30405 -1.875 0 Mabsadd L closepath F
.93333 .49588 -1.875 0 Mabsadd m
.93333 .49588 0 2.5 Mabsadd L
.93333 .49588 1.875 0 Mabsadd L
.93333 .49588 0 -2.5 Mabsadd L
.93333 .49588 -1.875 0 Mabsadd L closepath F
.93429 .47618 -1.875 0 Mabsadd m
.93429 .47618 0 2.5 Mabsadd L
.93429 .47618 1.875 0 Mabsadd L
.93429 .47618 0 -2.5 Mabsadd L
.93429 .47618 -1.875 0 Mabsadd L closepath F
.93524 .50229 -1.875 0 Mabsadd m
.93524 .50229 0 2.5 Mabsadd L
.93524 .50229 1.875 0 Mabsadd L
.93524 .50229 0 -2.5 Mabsadd L
.93524 .50229 -1.875 0 Mabsadd L closepath F
.93619 .50122 -1.875 0 Mabsadd m
.93619 .50122 0 2.5 Mabsadd L
.93619 .50122 1.875 0 Mabsadd L
.93619 .50122 0 -2.5 Mabsadd L
.93619 .50122 -1.875 0 Mabsadd L closepath F
.93714 .5028 -1.875 0 Mabsadd m
.93714 .5028 0 2.5 Mabsadd L
.93714 .5028 1.875 0 Mabsadd L
.93714 .5028 0 -2.5 Mabsadd L
.93714 .5028 -1.875 0 Mabsadd L closepath F
.9381 .51188 -1.875 0 Mabsadd m
.9381 .51188 0 2.5 Mabsadd L
.9381 .51188 1.875 0 Mabsadd L
.9381 .51188 0 -2.5 Mabsadd L
.9381 .51188 -1.875 0 Mabsadd L closepath F
.93905 .51192 -1.875 0 Mabsadd m
.93905 .51192 0 2.5 Mabsadd L
.93905 .51192 1.875 0 Mabsadd L
.93905 .51192 0 -2.5 Mabsadd L
.93905 .51192 -1.875 0 Mabsadd L closepath F
.94 .51203 -1.875 0 Mabsadd m
.94 .51203 0 2.5 Mabsadd L
.94 .51203 1.875 0 Mabsadd L
.94 .51203 0 -2.5 Mabsadd L
.94 .51203 -1.875 0 Mabsadd L closepath F
.94095 .51214 -1.875 0 Mabsadd m
.94095 .51214 0 2.5 Mabsadd L
.94095 .51214 1.875 0 Mabsadd L
.94095 .51214 0 -2.5 Mabsadd L
.94095 .51214 -1.875 0 Mabsadd L closepath F
.9419 .51222 -1.875 0 Mabsadd m
.9419 .51222 0 2.5 Mabsadd L
.9419 .51222 1.875 0 Mabsadd L
.9419 .51222 0 -2.5 Mabsadd L
.9419 .51222 -1.875 0 Mabsadd L closepath F
.94286 .51225 -1.875 0 Mabsadd m
.94286 .51225 0 2.5 Mabsadd L
.94286 .51225 1.875 0 Mabsadd L
.94286 .51225 0 -2.5 Mabsadd L
.94286 .51225 -1.875 0 Mabsadd L closepath F
.94381 .51226 -1.875 0 Mabsadd m
.94381 .51226 0 2.5 Mabsadd L
.94381 .51226 1.875 0 Mabsadd L
.94381 .51226 0 -2.5 Mabsadd L
.94381 .51226 -1.875 0 Mabsadd L closepath F
.94476 .51234 -1.875 0 Mabsadd m
.94476 .51234 0 2.5 Mabsadd L
.94476 .51234 1.875 0 Mabsadd L
.94476 .51234 0 -2.5 Mabsadd L
.94476 .51234 -1.875 0 Mabsadd L closepath F
.94571 .51256 -1.875 0 Mabsadd m
.94571 .51256 0 2.5 Mabsadd L
.94571 .51256 1.875 0 Mabsadd L
.94571 .51256 0 -2.5 Mabsadd L
.94571 .51256 -1.875 0 Mabsadd L closepath F
.94667 .51257 -1.875 0 Mabsadd m
.94667 .51257 0 2.5 Mabsadd L
.94667 .51257 1.875 0 Mabsadd L
.94667 .51257 0 -2.5 Mabsadd L
.94667 .51257 -1.875 0 Mabsadd L closepath F
.94762 .51261 -1.875 0 Mabsadd m
.94762 .51261 0 2.5 Mabsadd L
.94762 .51261 1.875 0 Mabsadd L
.94762 .51261 0 -2.5 Mabsadd L
.94762 .51261 -1.875 0 Mabsadd L closepath F
.94857 .51262 -1.875 0 Mabsadd m
.94857 .51262 0 2.5 Mabsadd L
.94857 .51262 1.875 0 Mabsadd L
.94857 .51262 0 -2.5 Mabsadd L
.94857 .51262 -1.875 0 Mabsadd L closepath F
.94952 .51263 -1.875 0 Mabsadd m
.94952 .51263 0 2.5 Mabsadd L
.94952 .51263 1.875 0 Mabsadd L
.94952 .51263 0 -2.5 Mabsadd L
.94952 .51263 -1.875 0 Mabsadd L closepath F
.95048 .51265 -1.875 0 Mabsadd m
.95048 .51265 0 2.5 Mabsadd L
.95048 .51265 1.875 0 Mabsadd L
.95048 .51265 0 -2.5 Mabsadd L
.95048 .51265 -1.875 0 Mabsadd L closepath F
.95143 .51265 -1.875 0 Mabsadd m
.95143 .51265 0 2.5 Mabsadd L
.95143 .51265 1.875 0 Mabsadd L
.95143 .51265 0 -2.5 Mabsadd L
.95143 .51265 -1.875 0 Mabsadd L closepath F
.95238 .51265 -1.875 0 Mabsadd m
.95238 .51265 0 2.5 Mabsadd L
.95238 .51265 1.875 0 Mabsadd L
.95238 .51265 0 -2.5 Mabsadd L
.95238 .51265 -1.875 0 Mabsadd L closepath F
.95333 .51265 -1.875 0 Mabsadd m
.95333 .51265 0 2.5 Mabsadd L
.95333 .51265 1.875 0 Mabsadd L
.95333 .51265 0 -2.5 Mabsadd L
.95333 .51265 -1.875 0 Mabsadd L closepath F
.95429 .51265 -1.875 0 Mabsadd m
.95429 .51265 0 2.5 Mabsadd L
.95429 .51265 1.875 0 Mabsadd L
.95429 .51265 0 -2.5 Mabsadd L
.95429 .51265 -1.875 0 Mabsadd L closepath F
.95524 .51266 -1.875 0 Mabsadd m
.95524 .51266 0 2.5 Mabsadd L
.95524 .51266 1.875 0 Mabsadd L
.95524 .51266 0 -2.5 Mabsadd L
.95524 .51266 -1.875 0 Mabsadd L closepath F
.95619 .51266 -1.875 0 Mabsadd m
.95619 .51266 0 2.5 Mabsadd L
.95619 .51266 1.875 0 Mabsadd L
.95619 .51266 0 -2.5 Mabsadd L
.95619 .51266 -1.875 0 Mabsadd L closepath F
.95714 .51266 -1.875 0 Mabsadd m
.95714 .51266 0 2.5 Mabsadd L
.95714 .51266 1.875 0 Mabsadd L
.95714 .51266 0 -2.5 Mabsadd L
.95714 .51266 -1.875 0 Mabsadd L closepath F
.9581 .51266 -1.875 0 Mabsadd m
.9581 .51266 0 2.5 Mabsadd L
.9581 .51266 1.875 0 Mabsadd L
.9581 .51266 0 -2.5 Mabsadd L
.9581 .51266 -1.875 0 Mabsadd L closepath F
.95905 .51266 -1.875 0 Mabsadd m
.95905 .51266 0 2.5 Mabsadd L
.95905 .51266 1.875 0 Mabsadd L
.95905 .51266 0 -2.5 Mabsadd L
.95905 .51266 -1.875 0 Mabsadd L closepath F
.96 .51266 -1.875 0 Mabsadd m
.96 .51266 0 2.5 Mabsadd L
.96 .51266 1.875 0 Mabsadd L
.96 .51266 0 -2.5 Mabsadd L
.96 .51266 -1.875 0 Mabsadd L closepath F
.96095 .51266 -1.875 0 Mabsadd m
.96095 .51266 0 2.5 Mabsadd L
.96095 .51266 1.875 0 Mabsadd L
.96095 .51266 0 -2.5 Mabsadd L
.96095 .51266 -1.875 0 Mabsadd L closepath F
.9619 .51266 -1.875 0 Mabsadd m
.9619 .51266 0 2.5 Mabsadd L
.9619 .51266 1.875 0 Mabsadd L
.9619 .51266 0 -2.5 Mabsadd L
.9619 .51266 -1.875 0 Mabsadd L closepath F
.96286 .51266 -1.875 0 Mabsadd m
.96286 .51266 0 2.5 Mabsadd L
.96286 .51266 1.875 0 Mabsadd L
.96286 .51266 0 -2.5 Mabsadd L
.96286 .51266 -1.875 0 Mabsadd L closepath F
.96381 .51265 -1.875 0 Mabsadd m
.96381 .51265 0 2.5 Mabsadd L
.96381 .51265 1.875 0 Mabsadd L
.96381 .51265 0 -2.5 Mabsadd L
.96381 .51265 -1.875 0 Mabsadd L closepath F
.96476 .51265 -1.875 0 Mabsadd m
.96476 .51265 0 2.5 Mabsadd L
.96476 .51265 1.875 0 Mabsadd L
.96476 .51265 0 -2.5 Mabsadd L
.96476 .51265 -1.875 0 Mabsadd L closepath F
.96571 .51265 -1.875 0 Mabsadd m
.96571 .51265 0 2.5 Mabsadd L
.96571 .51265 1.875 0 Mabsadd L
.96571 .51265 0 -2.5 Mabsadd L
.96571 .51265 -1.875 0 Mabsadd L closepath F
.96667 .51265 -1.875 0 Mabsadd m
.96667 .51265 0 2.5 Mabsadd L
.96667 .51265 1.875 0 Mabsadd L
.96667 .51265 0 -2.5 Mabsadd L
.96667 .51265 -1.875 0 Mabsadd L closepath F
.96762 .51265 -1.875 0 Mabsadd m
.96762 .51265 0 2.5 Mabsadd L
.96762 .51265 1.875 0 Mabsadd L
.96762 .51265 0 -2.5 Mabsadd L
.96762 .51265 -1.875 0 Mabsadd L closepath F
.96857 .51265 -1.875 0 Mabsadd m
.96857 .51265 0 2.5 Mabsadd L
.96857 .51265 1.875 0 Mabsadd L
.96857 .51265 0 -2.5 Mabsadd L
.96857 .51265 -1.875 0 Mabsadd L closepath F
.96952 .51264 -1.875 0 Mabsadd m
.96952 .51264 0 2.5 Mabsadd L
.96952 .51264 1.875 0 Mabsadd L
.96952 .51264 0 -2.5 Mabsadd L
.96952 .51264 -1.875 0 Mabsadd L closepath F
.97048 .51263 -1.875 0 Mabsadd m
.97048 .51263 0 2.5 Mabsadd L
.97048 .51263 1.875 0 Mabsadd L
.97048 .51263 0 -2.5 Mabsadd L
.97048 .51263 -1.875 0 Mabsadd L closepath F
.97143 .51263 -1.875 0 Mabsadd m
.97143 .51263 0 2.5 Mabsadd L
.97143 .51263 1.875 0 Mabsadd L
.97143 .51263 0 -2.5 Mabsadd L
.97143 .51263 -1.875 0 Mabsadd L closepath F
.97238 .51261 -1.875 0 Mabsadd m
.97238 .51261 0 2.5 Mabsadd L
.97238 .51261 1.875 0 Mabsadd L
.97238 .51261 0 -2.5 Mabsadd L
.97238 .51261 -1.875 0 Mabsadd L closepath F
.97333 .51255 -1.875 0 Mabsadd m
.97333 .51255 0 2.5 Mabsadd L
.97333 .51255 1.875 0 Mabsadd L
.97333 .51255 0 -2.5 Mabsadd L
.97333 .51255 -1.875 0 Mabsadd L closepath F
.97429 .51253 -1.875 0 Mabsadd m
.97429 .51253 0 2.5 Mabsadd L
.97429 .51253 1.875 0 Mabsadd L
.97429 .51253 0 -2.5 Mabsadd L
.97429 .51253 -1.875 0 Mabsadd L closepath F
.97524 .51237 -1.875 0 Mabsadd m
.97524 .51237 0 2.5 Mabsadd L
.97524 .51237 1.875 0 Mabsadd L
.97524 .51237 0 -2.5 Mabsadd L
.97524 .51237 -1.875 0 Mabsadd L closepath F
.97619 .51227 -1.875 0 Mabsadd m
.97619 .51227 0 2.5 Mabsadd L
.97619 .51227 1.875 0 Mabsadd L
.97619 .51227 0 -2.5 Mabsadd L
.97619 .51227 -1.875 0 Mabsadd L closepath F
0 0 1 r
.02381 .02381 0 2.5 Mabsadd m
.02381 .02381 -0.56518 .7779 Mabsadd L
.02381 .02381 -2.37764 .77254 Mabsadd L
.02381 .02381 -0.91448 -0.29713 Mabsadd L
.02381 .02381 -1.46946 -2.02254 Mabsadd L
.02381 .02381 0 -0.96154 Mabsadd L
.02381 .02381 1.46946 -2.02254 Mabsadd L
.02381 .02381 .91448 -0.29713 Mabsadd L
.02381 .02381 2.37764 .77254 Mabsadd L
.02381 .02381 .56518 .7779 Mabsadd L
.02381 .02381 0 2.5 Mabsadd L closepath F
.02476 .02505 0 2.5 Mabsadd m
.02476 .02505 -0.56518 .7779 Mabsadd L
.02476 .02505 -2.37764 .77254 Mabsadd L
.02476 .02505 -0.91448 -0.29713 Mabsadd L
.02476 .02505 -1.46946 -2.02254 Mabsadd L
.02476 .02505 0 -0.96154 Mabsadd L
.02476 .02505 1.46946 -2.02254 Mabsadd L
.02476 .02505 .91448 -0.29713 Mabsadd L
.02476 .02505 2.37764 .77254 Mabsadd L
.02476 .02505 .56518 .7779 Mabsadd L
.02476 .02505 0 2.5 Mabsadd L closepath F
.02571 .02488 0 2.5 Mabsadd m
.02571 .02488 -0.56518 .7779 Mabsadd L
.02571 .02488 -2.37764 .77254 Mabsadd L
.02571 .02488 -0.91448 -0.29713 Mabsadd L
.02571 .02488 -1.46946 -2.02254 Mabsadd L
.02571 .02488 0 -0.96154 Mabsadd L
.02571 .02488 1.46946 -2.02254 Mabsadd L
.02571 .02488 .91448 -0.29713 Mabsadd L
.02571 .02488 2.37764 .77254 Mabsadd L
.02571 .02488 .56518 .7779 Mabsadd L
.02571 .02488 0 2.5 Mabsadd L closepath F
.02667 .02502 0 2.5 Mabsadd m
.02667 .02502 -0.56518 .7779 Mabsadd L
.02667 .02502 -2.37764 .77254 Mabsadd L
.02667 .02502 -0.91448 -0.29713 Mabsadd L
.02667 .02502 -1.46946 -2.02254 Mabsadd L
.02667 .02502 0 -0.96154 Mabsadd L
.02667 .02502 1.46946 -2.02254 Mabsadd L
.02667 .02502 .91448 -0.29713 Mabsadd L
.02667 .02502 2.37764 .77254 Mabsadd L
.02667 .02502 .56518 .7779 Mabsadd L
.02667 .02502 0 2.5 Mabsadd L closepath F
.02762 .02659 0 2.5 Mabsadd m
.02762 .02659 -0.56518 .7779 Mabsadd L
.02762 .02659 -2.37764 .77254 Mabsadd L
.02762 .02659 -0.91448 -0.29713 Mabsadd L
.02762 .02659 -1.46946 -2.02254 Mabsadd L
.02762 .02659 0 -0.96154 Mabsadd L
.02762 .02659 1.46946 -2.02254 Mabsadd L
.02762 .02659 .91448 -0.29713 Mabsadd L
.02762 .02659 2.37764 .77254 Mabsadd L
.02762 .02659 .56518 .7779 Mabsadd L
.02762 .02659 0 2.5 Mabsadd L closepath F
.02857 .02475 0 2.5 Mabsadd m
.02857 .02475 -0.56518 .7779 Mabsadd L
.02857 .02475 -2.37764 .77254 Mabsadd L
.02857 .02475 -0.91448 -0.29713 Mabsadd L
.02857 .02475 -1.46946 -2.02254 Mabsadd L
.02857 .02475 0 -0.96154 Mabsadd L
.02857 .02475 1.46946 -2.02254 Mabsadd L
.02857 .02475 .91448 -0.29713 Mabsadd L
.02857 .02475 2.37764 .77254 Mabsadd L
.02857 .02475 .56518 .7779 Mabsadd L
.02857 .02475 0 2.5 Mabsadd L closepath F
.02952 .02834 0 2.5 Mabsadd m
.02952 .02834 -0.56518 .7779 Mabsadd L
.02952 .02834 -2.37764 .77254 Mabsadd L
.02952 .02834 -0.91448 -0.29713 Mabsadd L
.02952 .02834 -1.46946 -2.02254 Mabsadd L
.02952 .02834 0 -0.96154 Mabsadd L
.02952 .02834 1.46946 -2.02254 Mabsadd L
.02952 .02834 .91448 -0.29713 Mabsadd L
.02952 .02834 2.37764 .77254 Mabsadd L
.02952 .02834 .56518 .7779 Mabsadd L
.02952 .02834 0 2.5 Mabsadd L closepath F
.03048 .03111 0 2.5 Mabsadd m
.03048 .03111 -0.56518 .7779 Mabsadd L
.03048 .03111 -2.37764 .77254 Mabsadd L
.03048 .03111 -0.91448 -0.29713 Mabsadd L
.03048 .03111 -1.46946 -2.02254 Mabsadd L
.03048 .03111 0 -0.96154 Mabsadd L
.03048 .03111 1.46946 -2.02254 Mabsadd L
.03048 .03111 .91448 -0.29713 Mabsadd L
.03048 .03111 2.37764 .77254 Mabsadd L
.03048 .03111 .56518 .7779 Mabsadd L
.03048 .03111 0 2.5 Mabsadd L closepath F
.03143 .05286 0 2.5 Mabsadd m
.03143 .05286 -0.56518 .7779 Mabsadd L
.03143 .05286 -2.37764 .77254 Mabsadd L
.03143 .05286 -0.91448 -0.29713 Mabsadd L
.03143 .05286 -1.46946 -2.02254 Mabsadd L
.03143 .05286 0 -0.96154 Mabsadd L
.03143 .05286 1.46946 -2.02254 Mabsadd L
.03143 .05286 .91448 -0.29713 Mabsadd L
.03143 .05286 2.37764 .77254 Mabsadd L
.03143 .05286 .56518 .7779 Mabsadd L
.03143 .05286 0 2.5 Mabsadd L closepath F
.03238 .06032 0 2.5 Mabsadd m
.03238 .06032 -0.56518 .7779 Mabsadd L
.03238 .06032 -2.37764 .77254 Mabsadd L
.03238 .06032 -0.91448 -0.29713 Mabsadd L
.03238 .06032 -1.46946 -2.02254 Mabsadd L
.03238 .06032 0 -0.96154 Mabsadd L
.03238 .06032 1.46946 -2.02254 Mabsadd L
.03238 .06032 .91448 -0.29713 Mabsadd L
.03238 .06032 2.37764 .77254 Mabsadd L
.03238 .06032 .56518 .7779 Mabsadd L
.03238 .06032 0 2.5 Mabsadd L closepath F
.03333 .0628 0 2.5 Mabsadd m
.03333 .0628 -0.56518 .7779 Mabsadd L
.03333 .0628 -2.37764 .77254 Mabsadd L
.03333 .0628 -0.91448 -0.29713 Mabsadd L
.03333 .0628 -1.46946 -2.02254 Mabsadd L
.03333 .0628 0 -0.96154 Mabsadd L
.03333 .0628 1.46946 -2.02254 Mabsadd L
.03333 .0628 .91448 -0.29713 Mabsadd L
.03333 .0628 2.37764 .77254 Mabsadd L
.03333 .0628 .56518 .7779 Mabsadd L
.03333 .0628 0 2.5 Mabsadd L closepath F
.03429 .07004 0 2.5 Mabsadd m
.03429 .07004 -0.56518 .7779 Mabsadd L
.03429 .07004 -2.37764 .77254 Mabsadd L
.03429 .07004 -0.91448 -0.29713 Mabsadd L
.03429 .07004 -1.46946 -2.02254 Mabsadd L
.03429 .07004 0 -0.96154 Mabsadd L
.03429 .07004 1.46946 -2.02254 Mabsadd L
.03429 .07004 .91448 -0.29713 Mabsadd L
.03429 .07004 2.37764 .77254 Mabsadd L
.03429 .07004 .56518 .7779 Mabsadd L
.03429 .07004 0 2.5 Mabsadd L closepath F
.03524 .07039 0 2.5 Mabsadd m
.03524 .07039 -0.56518 .7779 Mabsadd L
.03524 .07039 -2.37764 .77254 Mabsadd L
.03524 .07039 -0.91448 -0.29713 Mabsadd L
.03524 .07039 -1.46946 -2.02254 Mabsadd L
.03524 .07039 0 -0.96154 Mabsadd L
.03524 .07039 1.46946 -2.02254 Mabsadd L
.03524 .07039 .91448 -0.29713 Mabsadd L
.03524 .07039 2.37764 .77254 Mabsadd L
.03524 .07039 .56518 .7779 Mabsadd L
.03524 .07039 0 2.5 Mabsadd L closepath F
.03619 .07149 0 2.5 Mabsadd m
.03619 .07149 -0.56518 .7779 Mabsadd L
.03619 .07149 -2.37764 .77254 Mabsadd L
.03619 .07149 -0.91448 -0.29713 Mabsadd L
.03619 .07149 -1.46946 -2.02254 Mabsadd L
.03619 .07149 0 -0.96154 Mabsadd L
.03619 .07149 1.46946 -2.02254 Mabsadd L
.03619 .07149 .91448 -0.29713 Mabsadd L
.03619 .07149 2.37764 .77254 Mabsadd L
.03619 .07149 .56518 .7779 Mabsadd L
.03619 .07149 0 2.5 Mabsadd L closepath F
.03714 .07156 0 2.5 Mabsadd m
.03714 .07156 -0.56518 .7779 Mabsadd L
.03714 .07156 -2.37764 .77254 Mabsadd L
.03714 .07156 -0.91448 -0.29713 Mabsadd L
.03714 .07156 -1.46946 -2.02254 Mabsadd L
.03714 .07156 0 -0.96154 Mabsadd L
.03714 .07156 1.46946 -2.02254 Mabsadd L
.03714 .07156 .91448 -0.29713 Mabsadd L
.03714 .07156 2.37764 .77254 Mabsadd L
.03714 .07156 .56518 .7779 Mabsadd L
.03714 .07156 0 2.5 Mabsadd L closepath F
.0381 .07178 0 2.5 Mabsadd m
.0381 .07178 -0.56518 .7779 Mabsadd L
.0381 .07178 -2.37764 .77254 Mabsadd L
.0381 .07178 -0.91448 -0.29713 Mabsadd L
.0381 .07178 -1.46946 -2.02254 Mabsadd L
.0381 .07178 0 -0.96154 Mabsadd L
.0381 .07178 1.46946 -2.02254 Mabsadd L
.0381 .07178 .91448 -0.29713 Mabsadd L
.0381 .07178 2.37764 .77254 Mabsadd L
.0381 .07178 .56518 .7779 Mabsadd L
.0381 .07178 0 2.5 Mabsadd L closepath F
.03905 .0718 0 2.5 Mabsadd m
.03905 .0718 -0.56518 .7779 Mabsadd L
.03905 .0718 -2.37764 .77254 Mabsadd L
.03905 .0718 -0.91448 -0.29713 Mabsadd L
.03905 .0718 -1.46946 -2.02254 Mabsadd L
.03905 .0718 0 -0.96154 Mabsadd L
.03905 .0718 1.46946 -2.02254 Mabsadd L
.03905 .0718 .91448 -0.29713 Mabsadd L
.03905 .0718 2.37764 .77254 Mabsadd L
.03905 .0718 .56518 .7779 Mabsadd L
.03905 .0718 0 2.5 Mabsadd L closepath F
.04 .07185 0 2.5 Mabsadd m
.04 .07185 -0.56518 .7779 Mabsadd L
.04 .07185 -2.37764 .77254 Mabsadd L
.04 .07185 -0.91448 -0.29713 Mabsadd L
.04 .07185 -1.46946 -2.02254 Mabsadd L
.04 .07185 0 -0.96154 Mabsadd L
.04 .07185 1.46946 -2.02254 Mabsadd L
.04 .07185 .91448 -0.29713 Mabsadd L
.04 .07185 2.37764 .77254 Mabsadd L
.04 .07185 .56518 .7779 Mabsadd L
.04 .07185 0 2.5 Mabsadd L closepath F
.04095 .07186 0 2.5 Mabsadd m
.04095 .07186 -0.56518 .7779 Mabsadd L
.04095 .07186 -2.37764 .77254 Mabsadd L
.04095 .07186 -0.91448 -0.29713 Mabsadd L
.04095 .07186 -1.46946 -2.02254 Mabsadd L
.04095 .07186 0 -0.96154 Mabsadd L
.04095 .07186 1.46946 -2.02254 Mabsadd L
.04095 .07186 .91448 -0.29713 Mabsadd L
.04095 .07186 2.37764 .77254 Mabsadd L
.04095 .07186 .56518 .7779 Mabsadd L
.04095 .07186 0 2.5 Mabsadd L closepath F
.0419 .07188 0 2.5 Mabsadd m
.0419 .07188 -0.56518 .7779 Mabsadd L
.0419 .07188 -2.37764 .77254 Mabsadd L
.0419 .07188 -0.91448 -0.29713 Mabsadd L
.0419 .07188 -1.46946 -2.02254 Mabsadd L
.0419 .07188 0 -0.96154 Mabsadd L
.0419 .07188 1.46946 -2.02254 Mabsadd L
.0419 .07188 .91448 -0.29713 Mabsadd L
.0419 .07188 2.37764 .77254 Mabsadd L
.0419 .07188 .56518 .7779 Mabsadd L
.0419 .07188 0 2.5 Mabsadd L closepath F
.04286 .07191 0 2.5 Mabsadd m
.04286 .07191 -0.56518 .7779 Mabsadd L
.04286 .07191 -2.37764 .77254 Mabsadd L
.04286 .07191 -0.91448 -0.29713 Mabsadd L
.04286 .07191 -1.46946 -2.02254 Mabsadd L
.04286 .07191 0 -0.96154 Mabsadd L
.04286 .07191 1.46946 -2.02254 Mabsadd L
.04286 .07191 .91448 -0.29713 Mabsadd L
.04286 .07191 2.37764 .77254 Mabsadd L
.04286 .07191 .56518 .7779 Mabsadd L
.04286 .07191 0 2.5 Mabsadd L closepath F
.04381 .07191 0 2.5 Mabsadd m
.04381 .07191 -0.56518 .7779 Mabsadd L
.04381 .07191 -2.37764 .77254 Mabsadd L
.04381 .07191 -0.91448 -0.29713 Mabsadd L
.04381 .07191 -1.46946 -2.02254 Mabsadd L
.04381 .07191 0 -0.96154 Mabsadd L
.04381 .07191 1.46946 -2.02254 Mabsadd L
.04381 .07191 .91448 -0.29713 Mabsadd L
.04381 .07191 2.37764 .77254 Mabsadd L
.04381 .07191 .56518 .7779 Mabsadd L
.04381 .07191 0 2.5 Mabsadd L closepath F
.04476 .07191 0 2.5 Mabsadd m
.04476 .07191 -0.56518 .7779 Mabsadd L
.04476 .07191 -2.37764 .77254 Mabsadd L
.04476 .07191 -0.91448 -0.29713 Mabsadd L
.04476 .07191 -1.46946 -2.02254 Mabsadd L
.04476 .07191 0 -0.96154 Mabsadd L
.04476 .07191 1.46946 -2.02254 Mabsadd L
.04476 .07191 .91448 -0.29713 Mabsadd L
.04476 .07191 2.37764 .77254 Mabsadd L
.04476 .07191 .56518 .7779 Mabsadd L
.04476 .07191 0 2.5 Mabsadd L closepath F
.04571 .07192 0 2.5 Mabsadd m
.04571 .07192 -0.56518 .7779 Mabsadd L
.04571 .07192 -2.37764 .77254 Mabsadd L
.04571 .07192 -0.91448 -0.29713 Mabsadd L
.04571 .07192 -1.46946 -2.02254 Mabsadd L
.04571 .07192 0 -0.96154 Mabsadd L
.04571 .07192 1.46946 -2.02254 Mabsadd L
.04571 .07192 .91448 -0.29713 Mabsadd L
.04571 .07192 2.37764 .77254 Mabsadd L
.04571 .07192 .56518 .7779 Mabsadd L
.04571 .07192 0 2.5 Mabsadd L closepath F
.04667 .07193 0 2.5 Mabsadd m
.04667 .07193 -0.56518 .7779 Mabsadd L
.04667 .07193 -2.37764 .77254 Mabsadd L
.04667 .07193 -0.91448 -0.29713 Mabsadd L
.04667 .07193 -1.46946 -2.02254 Mabsadd L
.04667 .07193 0 -0.96154 Mabsadd L
.04667 .07193 1.46946 -2.02254 Mabsadd L
.04667 .07193 .91448 -0.29713 Mabsadd L
.04667 .07193 2.37764 .77254 Mabsadd L
.04667 .07193 .56518 .7779 Mabsadd L
.04667 .07193 0 2.5 Mabsadd L closepath F
.04762 .07193 0 2.5 Mabsadd m
.04762 .07193 -0.56518 .7779 Mabsadd L
.04762 .07193 -2.37764 .77254 Mabsadd L
.04762 .07193 -0.91448 -0.29713 Mabsadd L
.04762 .07193 -1.46946 -2.02254 Mabsadd L
.04762 .07193 0 -0.96154 Mabsadd L
.04762 .07193 1.46946 -2.02254 Mabsadd L
.04762 .07193 .91448 -0.29713 Mabsadd L
.04762 .07193 2.37764 .77254 Mabsadd L
.04762 .07193 .56518 .7779 Mabsadd L
.04762 .07193 0 2.5 Mabsadd L closepath F
.04857 .07193 0 2.5 Mabsadd m
.04857 .07193 -0.56518 .7779 Mabsadd L
.04857 .07193 -2.37764 .77254 Mabsadd L
.04857 .07193 -0.91448 -0.29713 Mabsadd L
.04857 .07193 -1.46946 -2.02254 Mabsadd L
.04857 .07193 0 -0.96154 Mabsadd L
.04857 .07193 1.46946 -2.02254 Mabsadd L
.04857 .07193 .91448 -0.29713 Mabsadd L
.04857 .07193 2.37764 .77254 Mabsadd L
.04857 .07193 .56518 .7779 Mabsadd L
.04857 .07193 0 2.5 Mabsadd L closepath F
.04952 .07193 0 2.5 Mabsadd m
.04952 .07193 -0.56518 .7779 Mabsadd L
.04952 .07193 -2.37764 .77254 Mabsadd L
.04952 .07193 -0.91448 -0.29713 Mabsadd L
.04952 .07193 -1.46946 -2.02254 Mabsadd L
.04952 .07193 0 -0.96154 Mabsadd L
.04952 .07193 1.46946 -2.02254 Mabsadd L
.04952 .07193 .91448 -0.29713 Mabsadd L
.04952 .07193 2.37764 .77254 Mabsadd L
.04952 .07193 .56518 .7779 Mabsadd L
.04952 .07193 0 2.5 Mabsadd L closepath F
.05048 .07193 0 2.5 Mabsadd m
.05048 .07193 -0.56518 .7779 Mabsadd L
.05048 .07193 -2.37764 .77254 Mabsadd L
.05048 .07193 -0.91448 -0.29713 Mabsadd L
.05048 .07193 -1.46946 -2.02254 Mabsadd L
.05048 .07193 0 -0.96154 Mabsadd L
.05048 .07193 1.46946 -2.02254 Mabsadd L
.05048 .07193 .91448 -0.29713 Mabsadd L
.05048 .07193 2.37764 .77254 Mabsadd L
.05048 .07193 .56518 .7779 Mabsadd L
.05048 .07193 0 2.5 Mabsadd L closepath F
.05143 .07193 0 2.5 Mabsadd m
.05143 .07193 -0.56518 .7779 Mabsadd L
.05143 .07193 -2.37764 .77254 Mabsadd L
.05143 .07193 -0.91448 -0.29713 Mabsadd L
.05143 .07193 -1.46946 -2.02254 Mabsadd L
.05143 .07193 0 -0.96154 Mabsadd L
.05143 .07193 1.46946 -2.02254 Mabsadd L
.05143 .07193 .91448 -0.29713 Mabsadd L
.05143 .07193 2.37764 .77254 Mabsadd L
.05143 .07193 .56518 .7779 Mabsadd L
.05143 .07193 0 2.5 Mabsadd L closepath F
.05238 .07193 0 2.5 Mabsadd m
.05238 .07193 -0.56518 .7779 Mabsadd L
.05238 .07193 -2.37764 .77254 Mabsadd L
.05238 .07193 -0.91448 -0.29713 Mabsadd L
.05238 .07193 -1.46946 -2.02254 Mabsadd L
.05238 .07193 0 -0.96154 Mabsadd L
.05238 .07193 1.46946 -2.02254 Mabsadd L
.05238 .07193 .91448 -0.29713 Mabsadd L
.05238 .07193 2.37764 .77254 Mabsadd L
.05238 .07193 .56518 .7779 Mabsadd L
.05238 .07193 0 2.5 Mabsadd L closepath F
.05333 .07193 0 2.5 Mabsadd m
.05333 .07193 -0.56518 .7779 Mabsadd L
.05333 .07193 -2.37764 .77254 Mabsadd L
.05333 .07193 -0.91448 -0.29713 Mabsadd L
.05333 .07193 -1.46946 -2.02254 Mabsadd L
.05333 .07193 0 -0.96154 Mabsadd L
.05333 .07193 1.46946 -2.02254 Mabsadd L
.05333 .07193 .91448 -0.29713 Mabsadd L
.05333 .07193 2.37764 .77254 Mabsadd L
.05333 .07193 .56518 .7779 Mabsadd L
.05333 .07193 0 2.5 Mabsadd L closepath F
.05429 .07192 0 2.5 Mabsadd m
.05429 .07192 -0.56518 .7779 Mabsadd L
.05429 .07192 -2.37764 .77254 Mabsadd L
.05429 .07192 -0.91448 -0.29713 Mabsadd L
.05429 .07192 -1.46946 -2.02254 Mabsadd L
.05429 .07192 0 -0.96154 Mabsadd L
.05429 .07192 1.46946 -2.02254 Mabsadd L
.05429 .07192 .91448 -0.29713 Mabsadd L
.05429 .07192 2.37764 .77254 Mabsadd L
.05429 .07192 .56518 .7779 Mabsadd L
.05429 .07192 0 2.5 Mabsadd L closepath F
.05524 .07192 0 2.5 Mabsadd m
.05524 .07192 -0.56518 .7779 Mabsadd L
.05524 .07192 -2.37764 .77254 Mabsadd L
.05524 .07192 -0.91448 -0.29713 Mabsadd L
.05524 .07192 -1.46946 -2.02254 Mabsadd L
.05524 .07192 0 -0.96154 Mabsadd L
.05524 .07192 1.46946 -2.02254 Mabsadd L
.05524 .07192 .91448 -0.29713 Mabsadd L
.05524 .07192 2.37764 .77254 Mabsadd L
.05524 .07192 .56518 .7779 Mabsadd L
.05524 .07192 0 2.5 Mabsadd L closepath F
.05619 .07192 0 2.5 Mabsadd m
.05619 .07192 -0.56518 .7779 Mabsadd L
.05619 .07192 -2.37764 .77254 Mabsadd L
.05619 .07192 -0.91448 -0.29713 Mabsadd L
.05619 .07192 -1.46946 -2.02254 Mabsadd L
.05619 .07192 0 -0.96154 Mabsadd L
.05619 .07192 1.46946 -2.02254 Mabsadd L
.05619 .07192 .91448 -0.29713 Mabsadd L
.05619 .07192 2.37764 .77254 Mabsadd L
.05619 .07192 .56518 .7779 Mabsadd L
.05619 .07192 0 2.5 Mabsadd L closepath F
.05714 .07191 0 2.5 Mabsadd m
.05714 .07191 -0.56518 .7779 Mabsadd L
.05714 .07191 -2.37764 .77254 Mabsadd L
.05714 .07191 -0.91448 -0.29713 Mabsadd L
.05714 .07191 -1.46946 -2.02254 Mabsadd L
.05714 .07191 0 -0.96154 Mabsadd L
.05714 .07191 1.46946 -2.02254 Mabsadd L
.05714 .07191 .91448 -0.29713 Mabsadd L
.05714 .07191 2.37764 .77254 Mabsadd L
.05714 .07191 .56518 .7779 Mabsadd L
.05714 .07191 0 2.5 Mabsadd L closepath F
.0581 .0719 0 2.5 Mabsadd m
.0581 .0719 -0.56518 .7779 Mabsadd L
.0581 .0719 -2.37764 .77254 Mabsadd L
.0581 .0719 -0.91448 -0.29713 Mabsadd L
.0581 .0719 -1.46946 -2.02254 Mabsadd L
.0581 .0719 0 -0.96154 Mabsadd L
.0581 .0719 1.46946 -2.02254 Mabsadd L
.0581 .0719 .91448 -0.29713 Mabsadd L
.0581 .0719 2.37764 .77254 Mabsadd L
.0581 .0719 .56518 .7779 Mabsadd L
.0581 .0719 0 2.5 Mabsadd L closepath F
.05905 .0719 0 2.5 Mabsadd m
.05905 .0719 -0.56518 .7779 Mabsadd L
.05905 .0719 -2.37764 .77254 Mabsadd L
.05905 .0719 -0.91448 -0.29713 Mabsadd L
.05905 .0719 -1.46946 -2.02254 Mabsadd L
.05905 .0719 0 -0.96154 Mabsadd L
.05905 .0719 1.46946 -2.02254 Mabsadd L
.05905 .0719 .91448 -0.29713 Mabsadd L
.05905 .0719 2.37764 .77254 Mabsadd L
.05905 .0719 .56518 .7779 Mabsadd L
.05905 .0719 0 2.5 Mabsadd L closepath F
.06 .07189 0 2.5 Mabsadd m
.06 .07189 -0.56518 .7779 Mabsadd L
.06 .07189 -2.37764 .77254 Mabsadd L
.06 .07189 -0.91448 -0.29713 Mabsadd L
.06 .07189 -1.46946 -2.02254 Mabsadd L
.06 .07189 0 -0.96154 Mabsadd L
.06 .07189 1.46946 -2.02254 Mabsadd L
.06 .07189 .91448 -0.29713 Mabsadd L
.06 .07189 2.37764 .77254 Mabsadd L
.06 .07189 .56518 .7779 Mabsadd L
.06 .07189 0 2.5 Mabsadd L closepath F
.06095 .07187 0 2.5 Mabsadd m
.06095 .07187 -0.56518 .7779 Mabsadd L
.06095 .07187 -2.37764 .77254 Mabsadd L
.06095 .07187 -0.91448 -0.29713 Mabsadd L
.06095 .07187 -1.46946 -2.02254 Mabsadd L
.06095 .07187 0 -0.96154 Mabsadd L
.06095 .07187 1.46946 -2.02254 Mabsadd L
.06095 .07187 .91448 -0.29713 Mabsadd L
.06095 .07187 2.37764 .77254 Mabsadd L
.06095 .07187 .56518 .7779 Mabsadd L
.06095 .07187 0 2.5 Mabsadd L closepath F
.0619 .07187 0 2.5 Mabsadd m
.0619 .07187 -0.56518 .7779 Mabsadd L
.0619 .07187 -2.37764 .77254 Mabsadd L
.0619 .07187 -0.91448 -0.29713 Mabsadd L
.0619 .07187 -1.46946 -2.02254 Mabsadd L
.0619 .07187 0 -0.96154 Mabsadd L
.0619 .07187 1.46946 -2.02254 Mabsadd L
.0619 .07187 .91448 -0.29713 Mabsadd L
.0619 .07187 2.37764 .77254 Mabsadd L
.0619 .07187 .56518 .7779 Mabsadd L
.0619 .07187 0 2.5 Mabsadd L closepath F
.06286 .07183 0 2.5 Mabsadd m
.06286 .07183 -0.56518 .7779 Mabsadd L
.06286 .07183 -2.37764 .77254 Mabsadd L
.06286 .07183 -0.91448 -0.29713 Mabsadd L
.06286 .07183 -1.46946 -2.02254 Mabsadd L
.06286 .07183 0 -0.96154 Mabsadd L
.06286 .07183 1.46946 -2.02254 Mabsadd L
.06286 .07183 .91448 -0.29713 Mabsadd L
.06286 .07183 2.37764 .77254 Mabsadd L
.06286 .07183 .56518 .7779 Mabsadd L
.06286 .07183 0 2.5 Mabsadd L closepath F
.06381 .07182 0 2.5 Mabsadd m
.06381 .07182 -0.56518 .7779 Mabsadd L
.06381 .07182 -2.37764 .77254 Mabsadd L
.06381 .07182 -0.91448 -0.29713 Mabsadd L
.06381 .07182 -1.46946 -2.02254 Mabsadd L
.06381 .07182 0 -0.96154 Mabsadd L
.06381 .07182 1.46946 -2.02254 Mabsadd L
.06381 .07182 .91448 -0.29713 Mabsadd L
.06381 .07182 2.37764 .77254 Mabsadd L
.06381 .07182 .56518 .7779 Mabsadd L
.06381 .07182 0 2.5 Mabsadd L closepath F
.06476 .07169 0 2.5 Mabsadd m
.06476 .07169 -0.56518 .7779 Mabsadd L
.06476 .07169 -2.37764 .77254 Mabsadd L
.06476 .07169 -0.91448 -0.29713 Mabsadd L
.06476 .07169 -1.46946 -2.02254 Mabsadd L
.06476 .07169 0 -0.96154 Mabsadd L
.06476 .07169 1.46946 -2.02254 Mabsadd L
.06476 .07169 .91448 -0.29713 Mabsadd L
.06476 .07169 2.37764 .77254 Mabsadd L
.06476 .07169 .56518 .7779 Mabsadd L
.06476 .07169 0 2.5 Mabsadd L closepath F
.06571 .07164 0 2.5 Mabsadd m
.06571 .07164 -0.56518 .7779 Mabsadd L
.06571 .07164 -2.37764 .77254 Mabsadd L
.06571 .07164 -0.91448 -0.29713 Mabsadd L
.06571 .07164 -1.46946 -2.02254 Mabsadd L
.06571 .07164 0 -0.96154 Mabsadd L
.06571 .07164 1.46946 -2.02254 Mabsadd L
.06571 .07164 .91448 -0.29713 Mabsadd L
.06571 .07164 2.37764 .77254 Mabsadd L
.06571 .07164 .56518 .7779 Mabsadd L
.06571 .07164 0 2.5 Mabsadd L closepath F
.06667 .07108 0 2.5 Mabsadd m
.06667 .07108 -0.56518 .7779 Mabsadd L
.06667 .07108 -2.37764 .77254 Mabsadd L
.06667 .07108 -0.91448 -0.29713 Mabsadd L
.06667 .07108 -1.46946 -2.02254 Mabsadd L
.06667 .07108 0 -0.96154 Mabsadd L
.06667 .07108 1.46946 -2.02254 Mabsadd L
.06667 .07108 .91448 -0.29713 Mabsadd L
.06667 .07108 2.37764 .77254 Mabsadd L
.06667 .07108 .56518 .7779 Mabsadd L
.06667 .07108 0 2.5 Mabsadd L closepath F
.06762 .07077 0 2.5 Mabsadd m
.06762 .07077 -0.56518 .7779 Mabsadd L
.06762 .07077 -2.37764 .77254 Mabsadd L
.06762 .07077 -0.91448 -0.29713 Mabsadd L
.06762 .07077 -1.46946 -2.02254 Mabsadd L
.06762 .07077 0 -0.96154 Mabsadd L
.06762 .07077 1.46946 -2.02254 Mabsadd L
.06762 .07077 .91448 -0.29713 Mabsadd L
.06762 .07077 2.37764 .77254 Mabsadd L
.06762 .07077 .56518 .7779 Mabsadd L
.06762 .07077 0 2.5 Mabsadd L closepath F
.06857 .06748 0 2.5 Mabsadd m
.06857 .06748 -0.56518 .7779 Mabsadd L
.06857 .06748 -2.37764 .77254 Mabsadd L
.06857 .06748 -0.91448 -0.29713 Mabsadd L
.06857 .06748 -1.46946 -2.02254 Mabsadd L
.06857 .06748 0 -0.96154 Mabsadd L
.06857 .06748 1.46946 -2.02254 Mabsadd L
.06857 .06748 .91448 -0.29713 Mabsadd L
.06857 .06748 2.37764 .77254 Mabsadd L
.06857 .06748 .56518 .7779 Mabsadd L
.06857 .06748 0 2.5 Mabsadd L closepath F
.06952 .06522 0 2.5 Mabsadd m
.06952 .06522 -0.56518 .7779 Mabsadd L
.06952 .06522 -2.37764 .77254 Mabsadd L
.06952 .06522 -0.91448 -0.29713 Mabsadd L
.06952 .06522 -1.46946 -2.02254 Mabsadd L
.06952 .06522 0 -0.96154 Mabsadd L
.06952 .06522 1.46946 -2.02254 Mabsadd L
.06952 .06522 .91448 -0.29713 Mabsadd L
.06952 .06522 2.37764 .77254 Mabsadd L
.06952 .06522 .56518 .7779 Mabsadd L
.06952 .06522 0 2.5 Mabsadd L closepath F
.07048 .05815 0 2.5 Mabsadd m
.07048 .05815 -0.56518 .7779 Mabsadd L
.07048 .05815 -2.37764 .77254 Mabsadd L
.07048 .05815 -0.91448 -0.29713 Mabsadd L
.07048 .05815 -1.46946 -2.02254 Mabsadd L
.07048 .05815 0 -0.96154 Mabsadd L
.07048 .05815 1.46946 -2.02254 Mabsadd L
.07048 .05815 .91448 -0.29713 Mabsadd L
.07048 .05815 2.37764 .77254 Mabsadd L
.07048 .05815 .56518 .7779 Mabsadd L
.07048 .05815 0 2.5 Mabsadd L closepath F
.07143 .03363 0 2.5 Mabsadd m
.07143 .03363 -0.56518 .7779 Mabsadd L
.07143 .03363 -2.37764 .77254 Mabsadd L
.07143 .03363 -0.91448 -0.29713 Mabsadd L
.07143 .03363 -1.46946 -2.02254 Mabsadd L
.07143 .03363 0 -0.96154 Mabsadd L
.07143 .03363 1.46946 -2.02254 Mabsadd L
.07143 .03363 .91448 -0.29713 Mabsadd L
.07143 .03363 2.37764 .77254 Mabsadd L
.07143 .03363 .56518 .7779 Mabsadd L
.07143 .03363 0 2.5 Mabsadd L closepath F
.07238 .03364 0 2.5 Mabsadd m
.07238 .03364 -0.56518 .7779 Mabsadd L
.07238 .03364 -2.37764 .77254 Mabsadd L
.07238 .03364 -0.91448 -0.29713 Mabsadd L
.07238 .03364 -1.46946 -2.02254 Mabsadd L
.07238 .03364 0 -0.96154 Mabsadd L
.07238 .03364 1.46946 -2.02254 Mabsadd L
.07238 .03364 .91448 -0.29713 Mabsadd L
.07238 .03364 2.37764 .77254 Mabsadd L
.07238 .03364 .56518 .7779 Mabsadd L
.07238 .03364 0 2.5 Mabsadd L closepath F
.07333 .03438 0 2.5 Mabsadd m
.07333 .03438 -0.56518 .7779 Mabsadd L
.07333 .03438 -2.37764 .77254 Mabsadd L
.07333 .03438 -0.91448 -0.29713 Mabsadd L
.07333 .03438 -1.46946 -2.02254 Mabsadd L
.07333 .03438 0 -0.96154 Mabsadd L
.07333 .03438 1.46946 -2.02254 Mabsadd L
.07333 .03438 .91448 -0.29713 Mabsadd L
.07333 .03438 2.37764 .77254 Mabsadd L
.07333 .03438 .56518 .7779 Mabsadd L
.07333 .03438 0 2.5 Mabsadd L closepath F
.07429 .04906 0 2.5 Mabsadd m
.07429 .04906 -0.56518 .7779 Mabsadd L
.07429 .04906 -2.37764 .77254 Mabsadd L
.07429 .04906 -0.91448 -0.29713 Mabsadd L
.07429 .04906 -1.46946 -2.02254 Mabsadd L
.07429 .04906 0 -0.96154 Mabsadd L
.07429 .04906 1.46946 -2.02254 Mabsadd L
.07429 .04906 .91448 -0.29713 Mabsadd L
.07429 .04906 2.37764 .77254 Mabsadd L
.07429 .04906 .56518 .7779 Mabsadd L
.07429 .04906 0 2.5 Mabsadd L closepath F
.07524 .06604 0 2.5 Mabsadd m
.07524 .06604 -0.56518 .7779 Mabsadd L
.07524 .06604 -2.37764 .77254 Mabsadd L
.07524 .06604 -0.91448 -0.29713 Mabsadd L
.07524 .06604 -1.46946 -2.02254 Mabsadd L
.07524 .06604 0 -0.96154 Mabsadd L
.07524 .06604 1.46946 -2.02254 Mabsadd L
.07524 .06604 .91448 -0.29713 Mabsadd L
.07524 .06604 2.37764 .77254 Mabsadd L
.07524 .06604 .56518 .7779 Mabsadd L
.07524 .06604 0 2.5 Mabsadd L closepath F
.07619 .06711 0 2.5 Mabsadd m
.07619 .06711 -0.56518 .7779 Mabsadd L
.07619 .06711 -2.37764 .77254 Mabsadd L
.07619 .06711 -0.91448 -0.29713 Mabsadd L
.07619 .06711 -1.46946 -2.02254 Mabsadd L
.07619 .06711 0 -0.96154 Mabsadd L
.07619 .06711 1.46946 -2.02254 Mabsadd L
.07619 .06711 .91448 -0.29713 Mabsadd L
.07619 .06711 2.37764 .77254 Mabsadd L
.07619 .06711 .56518 .7779 Mabsadd L
.07619 .06711 0 2.5 Mabsadd L closepath F
.07714 .06861 0 2.5 Mabsadd m
.07714 .06861 -0.56518 .7779 Mabsadd L
.07714 .06861 -2.37764 .77254 Mabsadd L
.07714 .06861 -0.91448 -0.29713 Mabsadd L
.07714 .06861 -1.46946 -2.02254 Mabsadd L
.07714 .06861 0 -0.96154 Mabsadd L
.07714 .06861 1.46946 -2.02254 Mabsadd L
.07714 .06861 .91448 -0.29713 Mabsadd L
.07714 .06861 2.37764 .77254 Mabsadd L
.07714 .06861 .56518 .7779 Mabsadd L
.07714 .06861 0 2.5 Mabsadd L closepath F
.0781 .07268 0 2.5 Mabsadd m
.0781 .07268 -0.56518 .7779 Mabsadd L
.0781 .07268 -2.37764 .77254 Mabsadd L
.0781 .07268 -0.91448 -0.29713 Mabsadd L
.0781 .07268 -1.46946 -2.02254 Mabsadd L
.0781 .07268 0 -0.96154 Mabsadd L
.0781 .07268 1.46946 -2.02254 Mabsadd L
.0781 .07268 .91448 -0.29713 Mabsadd L
.0781 .07268 2.37764 .77254 Mabsadd L
.0781 .07268 .56518 .7779 Mabsadd L
.0781 .07268 0 2.5 Mabsadd L closepath F
.07905 .07319 0 2.5 Mabsadd m
.07905 .07319 -0.56518 .7779 Mabsadd L
.07905 .07319 -2.37764 .77254 Mabsadd L
.07905 .07319 -0.91448 -0.29713 Mabsadd L
.07905 .07319 -1.46946 -2.02254 Mabsadd L
.07905 .07319 0 -0.96154 Mabsadd L
.07905 .07319 1.46946 -2.02254 Mabsadd L
.07905 .07319 .91448 -0.29713 Mabsadd L
.07905 .07319 2.37764 .77254 Mabsadd L
.07905 .07319 .56518 .7779 Mabsadd L
.07905 .07319 0 2.5 Mabsadd L closepath F
.08 .07301 0 2.5 Mabsadd m
.08 .07301 -0.56518 .7779 Mabsadd L
.08 .07301 -2.37764 .77254 Mabsadd L
.08 .07301 -0.91448 -0.29713 Mabsadd L
.08 .07301 -1.46946 -2.02254 Mabsadd L
.08 .07301 0 -0.96154 Mabsadd L
.08 .07301 1.46946 -2.02254 Mabsadd L
.08 .07301 .91448 -0.29713 Mabsadd L
.08 .07301 2.37764 .77254 Mabsadd L
.08 .07301 .56518 .7779 Mabsadd L
.08 .07301 0 2.5 Mabsadd L closepath F
.08095 .07068 0 2.5 Mabsadd m
.08095 .07068 -0.56518 .7779 Mabsadd L
.08095 .07068 -2.37764 .77254 Mabsadd L
.08095 .07068 -0.91448 -0.29713 Mabsadd L
.08095 .07068 -1.46946 -2.02254 Mabsadd L
.08095 .07068 0 -0.96154 Mabsadd L
.08095 .07068 1.46946 -2.02254 Mabsadd L
.08095 .07068 .91448 -0.29713 Mabsadd L
.08095 .07068 2.37764 .77254 Mabsadd L
.08095 .07068 .56518 .7779 Mabsadd L
.08095 .07068 0 2.5 Mabsadd L closepath F
.0819 .07322 0 2.5 Mabsadd m
.0819 .07322 -0.56518 .7779 Mabsadd L
.0819 .07322 -2.37764 .77254 Mabsadd L
.0819 .07322 -0.91448 -0.29713 Mabsadd L
.0819 .07322 -1.46946 -2.02254 Mabsadd L
.0819 .07322 0 -0.96154 Mabsadd L
.0819 .07322 1.46946 -2.02254 Mabsadd L
.0819 .07322 .91448 -0.29713 Mabsadd L
.0819 .07322 2.37764 .77254 Mabsadd L
.0819 .07322 .56518 .7779 Mabsadd L
.0819 .07322 0 2.5 Mabsadd L closepath F
.08286 .0809 0 2.5 Mabsadd m
.08286 .0809 -0.56518 .7779 Mabsadd L
.08286 .0809 -2.37764 .77254 Mabsadd L
.08286 .0809 -0.91448 -0.29713 Mabsadd L
.08286 .0809 -1.46946 -2.02254 Mabsadd L
.08286 .0809 0 -0.96154 Mabsadd L
.08286 .0809 1.46946 -2.02254 Mabsadd L
.08286 .0809 .91448 -0.29713 Mabsadd L
.08286 .0809 2.37764 .77254 Mabsadd L
.08286 .0809 .56518 .7779 Mabsadd L
.08286 .0809 0 2.5 Mabsadd L closepath F
.08381 .08092 0 2.5 Mabsadd m
.08381 .08092 -0.56518 .7779 Mabsadd L
.08381 .08092 -2.37764 .77254 Mabsadd L
.08381 .08092 -0.91448 -0.29713 Mabsadd L
.08381 .08092 -1.46946 -2.02254 Mabsadd L
.08381 .08092 0 -0.96154 Mabsadd L
.08381 .08092 1.46946 -2.02254 Mabsadd L
.08381 .08092 .91448 -0.29713 Mabsadd L
.08381 .08092 2.37764 .77254 Mabsadd L
.08381 .08092 .56518 .7779 Mabsadd L
.08381 .08092 0 2.5 Mabsadd L closepath F
.08476 .08098 0 2.5 Mabsadd m
.08476 .08098 -0.56518 .7779 Mabsadd L
.08476 .08098 -2.37764 .77254 Mabsadd L
.08476 .08098 -0.91448 -0.29713 Mabsadd L
.08476 .08098 -1.46946 -2.02254 Mabsadd L
.08476 .08098 0 -0.96154 Mabsadd L
.08476 .08098 1.46946 -2.02254 Mabsadd L
.08476 .08098 .91448 -0.29713 Mabsadd L
.08476 .08098 2.37764 .77254 Mabsadd L
.08476 .08098 .56518 .7779 Mabsadd L
.08476 .08098 0 2.5 Mabsadd L closepath F
.08571 .08099 0 2.5 Mabsadd m
.08571 .08099 -0.56518 .7779 Mabsadd L
.08571 .08099 -2.37764 .77254 Mabsadd L
.08571 .08099 -0.91448 -0.29713 Mabsadd L
.08571 .08099 -1.46946 -2.02254 Mabsadd L
.08571 .08099 0 -0.96154 Mabsadd L
.08571 .08099 1.46946 -2.02254 Mabsadd L
.08571 .08099 .91448 -0.29713 Mabsadd L
.08571 .08099 2.37764 .77254 Mabsadd L
.08571 .08099 .56518 .7779 Mabsadd L
.08571 .08099 0 2.5 Mabsadd L closepath F
.08667 .08096 0 2.5 Mabsadd m
.08667 .08096 -0.56518 .7779 Mabsadd L
.08667 .08096 -2.37764 .77254 Mabsadd L
.08667 .08096 -0.91448 -0.29713 Mabsadd L
.08667 .08096 -1.46946 -2.02254 Mabsadd L
.08667 .08096 0 -0.96154 Mabsadd L
.08667 .08096 1.46946 -2.02254 Mabsadd L
.08667 .08096 .91448 -0.29713 Mabsadd L
.08667 .08096 2.37764 .77254 Mabsadd L
.08667 .08096 .56518 .7779 Mabsadd L
.08667 .08096 0 2.5 Mabsadd L closepath F
.08762 .0809 0 2.5 Mabsadd m
.08762 .0809 -0.56518 .7779 Mabsadd L
.08762 .0809 -2.37764 .77254 Mabsadd L
.08762 .0809 -0.91448 -0.29713 Mabsadd L
.08762 .0809 -1.46946 -2.02254 Mabsadd L
.08762 .0809 0 -0.96154 Mabsadd L
.08762 .0809 1.46946 -2.02254 Mabsadd L
.08762 .0809 .91448 -0.29713 Mabsadd L
.08762 .0809 2.37764 .77254 Mabsadd L
.08762 .0809 .56518 .7779 Mabsadd L
.08762 .0809 0 2.5 Mabsadd L closepath F
.08857 .08129 0 2.5 Mabsadd m
.08857 .08129 -0.56518 .7779 Mabsadd L
.08857 .08129 -2.37764 .77254 Mabsadd L
.08857 .08129 -0.91448 -0.29713 Mabsadd L
.08857 .08129 -1.46946 -2.02254 Mabsadd L
.08857 .08129 0 -0.96154 Mabsadd L
.08857 .08129 1.46946 -2.02254 Mabsadd L
.08857 .08129 .91448 -0.29713 Mabsadd L
.08857 .08129 2.37764 .77254 Mabsadd L
.08857 .08129 .56518 .7779 Mabsadd L
.08857 .08129 0 2.5 Mabsadd L closepath F
.08952 .08253 0 2.5 Mabsadd m
.08952 .08253 -0.56518 .7779 Mabsadd L
.08952 .08253 -2.37764 .77254 Mabsadd L
.08952 .08253 -0.91448 -0.29713 Mabsadd L
.08952 .08253 -1.46946 -2.02254 Mabsadd L
.08952 .08253 0 -0.96154 Mabsadd L
.08952 .08253 1.46946 -2.02254 Mabsadd L
.08952 .08253 .91448 -0.29713 Mabsadd L
.08952 .08253 2.37764 .77254 Mabsadd L
.08952 .08253 .56518 .7779 Mabsadd L
.08952 .08253 0 2.5 Mabsadd L closepath F
.09048 .0826 0 2.5 Mabsadd m
.09048 .0826 -0.56518 .7779 Mabsadd L
.09048 .0826 -2.37764 .77254 Mabsadd L
.09048 .0826 -0.91448 -0.29713 Mabsadd L
.09048 .0826 -1.46946 -2.02254 Mabsadd L
.09048 .0826 0 -0.96154 Mabsadd L
.09048 .0826 1.46946 -2.02254 Mabsadd L
.09048 .0826 .91448 -0.29713 Mabsadd L
.09048 .0826 2.37764 .77254 Mabsadd L
.09048 .0826 .56518 .7779 Mabsadd L
.09048 .0826 0 2.5 Mabsadd L closepath F
.09143 .08284 0 2.5 Mabsadd m
.09143 .08284 -0.56518 .7779 Mabsadd L
.09143 .08284 -2.37764 .77254 Mabsadd L
.09143 .08284 -0.91448 -0.29713 Mabsadd L
.09143 .08284 -1.46946 -2.02254 Mabsadd L
.09143 .08284 0 -0.96154 Mabsadd L
.09143 .08284 1.46946 -2.02254 Mabsadd L
.09143 .08284 .91448 -0.29713 Mabsadd L
.09143 .08284 2.37764 .77254 Mabsadd L
.09143 .08284 .56518 .7779 Mabsadd L
.09143 .08284 0 2.5 Mabsadd L closepath F
.09238 .08286 0 2.5 Mabsadd m
.09238 .08286 -0.56518 .7779 Mabsadd L
.09238 .08286 -2.37764 .77254 Mabsadd L
.09238 .08286 -0.91448 -0.29713 Mabsadd L
.09238 .08286 -1.46946 -2.02254 Mabsadd L
.09238 .08286 0 -0.96154 Mabsadd L
.09238 .08286 1.46946 -2.02254 Mabsadd L
.09238 .08286 .91448 -0.29713 Mabsadd L
.09238 .08286 2.37764 .77254 Mabsadd L
.09238 .08286 .56518 .7779 Mabsadd L
.09238 .08286 0 2.5 Mabsadd L closepath F
.09333 .08292 0 2.5 Mabsadd m
.09333 .08292 -0.56518 .7779 Mabsadd L
.09333 .08292 -2.37764 .77254 Mabsadd L
.09333 .08292 -0.91448 -0.29713 Mabsadd L
.09333 .08292 -1.46946 -2.02254 Mabsadd L
.09333 .08292 0 -0.96154 Mabsadd L
.09333 .08292 1.46946 -2.02254 Mabsadd L
.09333 .08292 .91448 -0.29713 Mabsadd L
.09333 .08292 2.37764 .77254 Mabsadd L
.09333 .08292 .56518 .7779 Mabsadd L
.09333 .08292 0 2.5 Mabsadd L closepath F
.09429 .08293 0 2.5 Mabsadd m
.09429 .08293 -0.56518 .7779 Mabsadd L
.09429 .08293 -2.37764 .77254 Mabsadd L
.09429 .08293 -0.91448 -0.29713 Mabsadd L
.09429 .08293 -1.46946 -2.02254 Mabsadd L
.09429 .08293 0 -0.96154 Mabsadd L
.09429 .08293 1.46946 -2.02254 Mabsadd L
.09429 .08293 .91448 -0.29713 Mabsadd L
.09429 .08293 2.37764 .77254 Mabsadd L
.09429 .08293 .56518 .7779 Mabsadd L
.09429 .08293 0 2.5 Mabsadd L closepath F
.09524 .08295 0 2.5 Mabsadd m
.09524 .08295 -0.56518 .7779 Mabsadd L
.09524 .08295 -2.37764 .77254 Mabsadd L
.09524 .08295 -0.91448 -0.29713 Mabsadd L
.09524 .08295 -1.46946 -2.02254 Mabsadd L
.09524 .08295 0 -0.96154 Mabsadd L
.09524 .08295 1.46946 -2.02254 Mabsadd L
.09524 .08295 .91448 -0.29713 Mabsadd L
.09524 .08295 2.37764 .77254 Mabsadd L
.09524 .08295 .56518 .7779 Mabsadd L
.09524 .08295 0 2.5 Mabsadd L closepath F
.09619 .08298 0 2.5 Mabsadd m
.09619 .08298 -0.56518 .7779 Mabsadd L
.09619 .08298 -2.37764 .77254 Mabsadd L
.09619 .08298 -0.91448 -0.29713 Mabsadd L
.09619 .08298 -1.46946 -2.02254 Mabsadd L
.09619 .08298 0 -0.96154 Mabsadd L
.09619 .08298 1.46946 -2.02254 Mabsadd L
.09619 .08298 .91448 -0.29713 Mabsadd L
.09619 .08298 2.37764 .77254 Mabsadd L
.09619 .08298 .56518 .7779 Mabsadd L
.09619 .08298 0 2.5 Mabsadd L closepath F
.09714 .08298 0 2.5 Mabsadd m
.09714 .08298 -0.56518 .7779 Mabsadd L
.09714 .08298 -2.37764 .77254 Mabsadd L
.09714 .08298 -0.91448 -0.29713 Mabsadd L
.09714 .08298 -1.46946 -2.02254 Mabsadd L
.09714 .08298 0 -0.96154 Mabsadd L
.09714 .08298 1.46946 -2.02254 Mabsadd L
.09714 .08298 .91448 -0.29713 Mabsadd L
.09714 .08298 2.37764 .77254 Mabsadd L
.09714 .08298 .56518 .7779 Mabsadd L
.09714 .08298 0 2.5 Mabsadd L closepath F
.0981 .08298 0 2.5 Mabsadd m
.0981 .08298 -0.56518 .7779 Mabsadd L
.0981 .08298 -2.37764 .77254 Mabsadd L
.0981 .08298 -0.91448 -0.29713 Mabsadd L
.0981 .08298 -1.46946 -2.02254 Mabsadd L
.0981 .08298 0 -0.96154 Mabsadd L
.0981 .08298 1.46946 -2.02254 Mabsadd L
.0981 .08298 .91448 -0.29713 Mabsadd L
.0981 .08298 2.37764 .77254 Mabsadd L
.0981 .08298 .56518 .7779 Mabsadd L
.0981 .08298 0 2.5 Mabsadd L closepath F
.09905 .08299 0 2.5 Mabsadd m
.09905 .08299 -0.56518 .7779 Mabsadd L
.09905 .08299 -2.37764 .77254 Mabsadd L
.09905 .08299 -0.91448 -0.29713 Mabsadd L
.09905 .08299 -1.46946 -2.02254 Mabsadd L
.09905 .08299 0 -0.96154 Mabsadd L
.09905 .08299 1.46946 -2.02254 Mabsadd L
.09905 .08299 .91448 -0.29713 Mabsadd L
.09905 .08299 2.37764 .77254 Mabsadd L
.09905 .08299 .56518 .7779 Mabsadd L
.09905 .08299 0 2.5 Mabsadd L closepath F
.1 .083 0 2.5 Mabsadd m
.1 .083 -0.56518 .7779 Mabsadd L
.1 .083 -2.37764 .77254 Mabsadd L
.1 .083 -0.91448 -0.29713 Mabsadd L
.1 .083 -1.46946 -2.02254 Mabsadd L
.1 .083 0 -0.96154 Mabsadd L
.1 .083 1.46946 -2.02254 Mabsadd L
.1 .083 .91448 -0.29713 Mabsadd L
.1 .083 2.37764 .77254 Mabsadd L
.1 .083 .56518 .7779 Mabsadd L
.1 .083 0 2.5 Mabsadd L closepath F
.10095 .083 0 2.5 Mabsadd m
.10095 .083 -0.56518 .7779 Mabsadd L
.10095 .083 -2.37764 .77254 Mabsadd L
.10095 .083 -0.91448 -0.29713 Mabsadd L
.10095 .083 -1.46946 -2.02254 Mabsadd L
.10095 .083 0 -0.96154 Mabsadd L
.10095 .083 1.46946 -2.02254 Mabsadd L
.10095 .083 .91448 -0.29713 Mabsadd L
.10095 .083 2.37764 .77254 Mabsadd L
.10095 .083 .56518 .7779 Mabsadd L
.10095 .083 0 2.5 Mabsadd L closepath F
.1019 .083 0 2.5 Mabsadd m
.1019 .083 -0.56518 .7779 Mabsadd L
.1019 .083 -2.37764 .77254 Mabsadd L
.1019 .083 -0.91448 -0.29713 Mabsadd L
.1019 .083 -1.46946 -2.02254 Mabsadd L
.1019 .083 0 -0.96154 Mabsadd L
.1019 .083 1.46946 -2.02254 Mabsadd L
.1019 .083 .91448 -0.29713 Mabsadd L
.1019 .083 2.37764 .77254 Mabsadd L
.1019 .083 .56518 .7779 Mabsadd L
.1019 .083 0 2.5 Mabsadd L closepath F
.10286 .083 0 2.5 Mabsadd m
.10286 .083 -0.56518 .7779 Mabsadd L
.10286 .083 -2.37764 .77254 Mabsadd L
.10286 .083 -0.91448 -0.29713 Mabsadd L
.10286 .083 -1.46946 -2.02254 Mabsadd L
.10286 .083 0 -0.96154 Mabsadd L
.10286 .083 1.46946 -2.02254 Mabsadd L
.10286 .083 .91448 -0.29713 Mabsadd L
.10286 .083 2.37764 .77254 Mabsadd L
.10286 .083 .56518 .7779 Mabsadd L
.10286 .083 0 2.5 Mabsadd L closepath F
.10381 .083 0 2.5 Mabsadd m
.10381 .083 -0.56518 .7779 Mabsadd L
.10381 .083 -2.37764 .77254 Mabsadd L
.10381 .083 -0.91448 -0.29713 Mabsadd L
.10381 .083 -1.46946 -2.02254 Mabsadd L
.10381 .083 0 -0.96154 Mabsadd L
.10381 .083 1.46946 -2.02254 Mabsadd L
.10381 .083 .91448 -0.29713 Mabsadd L
.10381 .083 2.37764 .77254 Mabsadd L
.10381 .083 .56518 .7779 Mabsadd L
.10381 .083 0 2.5 Mabsadd L closepath F
.10476 .083 0 2.5 Mabsadd m
.10476 .083 -0.56518 .7779 Mabsadd L
.10476 .083 -2.37764 .77254 Mabsadd L
.10476 .083 -0.91448 -0.29713 Mabsadd L
.10476 .083 -1.46946 -2.02254 Mabsadd L
.10476 .083 0 -0.96154 Mabsadd L
.10476 .083 1.46946 -2.02254 Mabsadd L
.10476 .083 .91448 -0.29713 Mabsadd L
.10476 .083 2.37764 .77254 Mabsadd L
.10476 .083 .56518 .7779 Mabsadd L
.10476 .083 0 2.5 Mabsadd L closepath F
.10571 .083 0 2.5 Mabsadd m
.10571 .083 -0.56518 .7779 Mabsadd L
.10571 .083 -2.37764 .77254 Mabsadd L
.10571 .083 -0.91448 -0.29713 Mabsadd L
.10571 .083 -1.46946 -2.02254 Mabsadd L
.10571 .083 0 -0.96154 Mabsadd L
.10571 .083 1.46946 -2.02254 Mabsadd L
.10571 .083 .91448 -0.29713 Mabsadd L
.10571 .083 2.37764 .77254 Mabsadd L
.10571 .083 .56518 .7779 Mabsadd L
.10571 .083 0 2.5 Mabsadd L closepath F
.10667 .083 0 2.5 Mabsadd m
.10667 .083 -0.56518 .7779 Mabsadd L
.10667 .083 -2.37764 .77254 Mabsadd L
.10667 .083 -0.91448 -0.29713 Mabsadd L
.10667 .083 -1.46946 -2.02254 Mabsadd L
.10667 .083 0 -0.96154 Mabsadd L
.10667 .083 1.46946 -2.02254 Mabsadd L
.10667 .083 .91448 -0.29713 Mabsadd L
.10667 .083 2.37764 .77254 Mabsadd L
.10667 .083 .56518 .7779 Mabsadd L
.10667 .083 0 2.5 Mabsadd L closepath F
.10762 .08299 0 2.5 Mabsadd m
.10762 .08299 -0.56518 .7779 Mabsadd L
.10762 .08299 -2.37764 .77254 Mabsadd L
.10762 .08299 -0.91448 -0.29713 Mabsadd L
.10762 .08299 -1.46946 -2.02254 Mabsadd L
.10762 .08299 0 -0.96154 Mabsadd L
.10762 .08299 1.46946 -2.02254 Mabsadd L
.10762 .08299 .91448 -0.29713 Mabsadd L
.10762 .08299 2.37764 .77254 Mabsadd L
.10762 .08299 .56518 .7779 Mabsadd L
.10762 .08299 0 2.5 Mabsadd L closepath F
.10857 .08299 0 2.5 Mabsadd m
.10857 .08299 -0.56518 .7779 Mabsadd L
.10857 .08299 -2.37764 .77254 Mabsadd L
.10857 .08299 -0.91448 -0.29713 Mabsadd L
.10857 .08299 -1.46946 -2.02254 Mabsadd L
.10857 .08299 0 -0.96154 Mabsadd L
.10857 .08299 1.46946 -2.02254 Mabsadd L
.10857 .08299 .91448 -0.29713 Mabsadd L
.10857 .08299 2.37764 .77254 Mabsadd L
.10857 .08299 .56518 .7779 Mabsadd L
.10857 .08299 0 2.5 Mabsadd L closepath F
.10952 .08299 0 2.5 Mabsadd m
.10952 .08299 -0.56518 .7779 Mabsadd L
.10952 .08299 -2.37764 .77254 Mabsadd L
.10952 .08299 -0.91448 -0.29713 Mabsadd L
.10952 .08299 -1.46946 -2.02254 Mabsadd L
.10952 .08299 0 -0.96154 Mabsadd L
.10952 .08299 1.46946 -2.02254 Mabsadd L
.10952 .08299 .91448 -0.29713 Mabsadd L
.10952 .08299 2.37764 .77254 Mabsadd L
.10952 .08299 .56518 .7779 Mabsadd L
.10952 .08299 0 2.5 Mabsadd L closepath F
.11048 .08299 0 2.5 Mabsadd m
.11048 .08299 -0.56518 .7779 Mabsadd L
.11048 .08299 -2.37764 .77254 Mabsadd L
.11048 .08299 -0.91448 -0.29713 Mabsadd L
.11048 .08299 -1.46946 -2.02254 Mabsadd L
.11048 .08299 0 -0.96154 Mabsadd L
.11048 .08299 1.46946 -2.02254 Mabsadd L
.11048 .08299 .91448 -0.29713 Mabsadd L
.11048 .08299 2.37764 .77254 Mabsadd L
.11048 .08299 .56518 .7779 Mabsadd L
.11048 .08299 0 2.5 Mabsadd L closepath F
.11143 .08298 0 2.5 Mabsadd m
.11143 .08298 -0.56518 .7779 Mabsadd L
.11143 .08298 -2.37764 .77254 Mabsadd L
.11143 .08298 -0.91448 -0.29713 Mabsadd L
.11143 .08298 -1.46946 -2.02254 Mabsadd L
.11143 .08298 0 -0.96154 Mabsadd L
.11143 .08298 1.46946 -2.02254 Mabsadd L
.11143 .08298 .91448 -0.29713 Mabsadd L
.11143 .08298 2.37764 .77254 Mabsadd L
.11143 .08298 .56518 .7779 Mabsadd L
.11143 .08298 0 2.5 Mabsadd L closepath F
.11238 .08298 0 2.5 Mabsadd m
.11238 .08298 -0.56518 .7779 Mabsadd L
.11238 .08298 -2.37764 .77254 Mabsadd L
.11238 .08298 -0.91448 -0.29713 Mabsadd L
.11238 .08298 -1.46946 -2.02254 Mabsadd L
.11238 .08298 0 -0.96154 Mabsadd L
.11238 .08298 1.46946 -2.02254 Mabsadd L
.11238 .08298 .91448 -0.29713 Mabsadd L
.11238 .08298 2.37764 .77254 Mabsadd L
.11238 .08298 .56518 .7779 Mabsadd L
.11238 .08298 0 2.5 Mabsadd L closepath F
.11333 .08297 0 2.5 Mabsadd m
.11333 .08297 -0.56518 .7779 Mabsadd L
.11333 .08297 -2.37764 .77254 Mabsadd L
.11333 .08297 -0.91448 -0.29713 Mabsadd L
.11333 .08297 -1.46946 -2.02254 Mabsadd L
.11333 .08297 0 -0.96154 Mabsadd L
.11333 .08297 1.46946 -2.02254 Mabsadd L
.11333 .08297 .91448 -0.29713 Mabsadd L
.11333 .08297 2.37764 .77254 Mabsadd L
.11333 .08297 .56518 .7779 Mabsadd L
.11333 .08297 0 2.5 Mabsadd L closepath F
.11429 .08295 0 2.5 Mabsadd m
.11429 .08295 -0.56518 .7779 Mabsadd L
.11429 .08295 -2.37764 .77254 Mabsadd L
.11429 .08295 -0.91448 -0.29713 Mabsadd L
.11429 .08295 -1.46946 -2.02254 Mabsadd L
.11429 .08295 0 -0.96154 Mabsadd L
.11429 .08295 1.46946 -2.02254 Mabsadd L
.11429 .08295 .91448 -0.29713 Mabsadd L
.11429 .08295 2.37764 .77254 Mabsadd L
.11429 .08295 .56518 .7779 Mabsadd L
.11429 .08295 0 2.5 Mabsadd L closepath F
.11524 .08294 0 2.5 Mabsadd m
.11524 .08294 -0.56518 .7779 Mabsadd L
.11524 .08294 -2.37764 .77254 Mabsadd L
.11524 .08294 -0.91448 -0.29713 Mabsadd L
.11524 .08294 -1.46946 -2.02254 Mabsadd L
.11524 .08294 0 -0.96154 Mabsadd L
.11524 .08294 1.46946 -2.02254 Mabsadd L
.11524 .08294 .91448 -0.29713 Mabsadd L
.11524 .08294 2.37764 .77254 Mabsadd L
.11524 .08294 .56518 .7779 Mabsadd L
.11524 .08294 0 2.5 Mabsadd L closepath F
.11619 .08291 0 2.5 Mabsadd m
.11619 .08291 -0.56518 .7779 Mabsadd L
.11619 .08291 -2.37764 .77254 Mabsadd L
.11619 .08291 -0.91448 -0.29713 Mabsadd L
.11619 .08291 -1.46946 -2.02254 Mabsadd L
.11619 .08291 0 -0.96154 Mabsadd L
.11619 .08291 1.46946 -2.02254 Mabsadd L
.11619 .08291 .91448 -0.29713 Mabsadd L
.11619 .08291 2.37764 .77254 Mabsadd L
.11619 .08291 .56518 .7779 Mabsadd L
.11619 .08291 0 2.5 Mabsadd L closepath F
.11714 .0829 0 2.5 Mabsadd m
.11714 .0829 -0.56518 .7779 Mabsadd L
.11714 .0829 -2.37764 .77254 Mabsadd L
.11714 .0829 -0.91448 -0.29713 Mabsadd L
.11714 .0829 -1.46946 -2.02254 Mabsadd L
.11714 .0829 0 -0.96154 Mabsadd L
.11714 .0829 1.46946 -2.02254 Mabsadd L
.11714 .0829 .91448 -0.29713 Mabsadd L
.11714 .0829 2.37764 .77254 Mabsadd L
.11714 .0829 .56518 .7779 Mabsadd L
.11714 .0829 0 2.5 Mabsadd L closepath F
.1181 .08279 0 2.5 Mabsadd m
.1181 .08279 -0.56518 .7779 Mabsadd L
.1181 .08279 -2.37764 .77254 Mabsadd L
.1181 .08279 -0.91448 -0.29713 Mabsadd L
.1181 .08279 -1.46946 -2.02254 Mabsadd L
.1181 .08279 0 -0.96154 Mabsadd L
.1181 .08279 1.46946 -2.02254 Mabsadd L
.1181 .08279 .91448 -0.29713 Mabsadd L
.1181 .08279 2.37764 .77254 Mabsadd L
.1181 .08279 .56518 .7779 Mabsadd L
.1181 .08279 0 2.5 Mabsadd L closepath F
.11905 .08274 0 2.5 Mabsadd m
.11905 .08274 -0.56518 .7779 Mabsadd L
.11905 .08274 -2.37764 .77254 Mabsadd L
.11905 .08274 -0.91448 -0.29713 Mabsadd L
.11905 .08274 -1.46946 -2.02254 Mabsadd L
.11905 .08274 0 -0.96154 Mabsadd L
.11905 .08274 1.46946 -2.02254 Mabsadd L
.11905 .08274 .91448 -0.29713 Mabsadd L
.11905 .08274 2.37764 .77254 Mabsadd L
.11905 .08274 .56518 .7779 Mabsadd L
.11905 .08274 0 2.5 Mabsadd L closepath F
.12 .08223 0 2.5 Mabsadd m
.12 .08223 -0.56518 .7779 Mabsadd L
.12 .08223 -2.37764 .77254 Mabsadd L
.12 .08223 -0.91448 -0.29713 Mabsadd L
.12 .08223 -1.46946 -2.02254 Mabsadd L
.12 .08223 0 -0.96154 Mabsadd L
.12 .08223 1.46946 -2.02254 Mabsadd L
.12 .08223 .91448 -0.29713 Mabsadd L
.12 .08223 2.37764 .77254 Mabsadd L
.12 .08223 .56518 .7779 Mabsadd L
.12 .08223 0 2.5 Mabsadd L closepath F
.12095 .08196 0 2.5 Mabsadd m
.12095 .08196 -0.56518 .7779 Mabsadd L
.12095 .08196 -2.37764 .77254 Mabsadd L
.12095 .08196 -0.91448 -0.29713 Mabsadd L
.12095 .08196 -1.46946 -2.02254 Mabsadd L
.12095 .08196 0 -0.96154 Mabsadd L
.12095 .08196 1.46946 -2.02254 Mabsadd L
.12095 .08196 .91448 -0.29713 Mabsadd L
.12095 .08196 2.37764 .77254 Mabsadd L
.12095 .08196 .56518 .7779 Mabsadd L
.12095 .08196 0 2.5 Mabsadd L closepath F
.1219 .07905 0 2.5 Mabsadd m
.1219 .07905 -0.56518 .7779 Mabsadd L
.1219 .07905 -2.37764 .77254 Mabsadd L
.1219 .07905 -0.91448 -0.29713 Mabsadd L
.1219 .07905 -1.46946 -2.02254 Mabsadd L
.1219 .07905 0 -0.96154 Mabsadd L
.1219 .07905 1.46946 -2.02254 Mabsadd L
.1219 .07905 .91448 -0.29713 Mabsadd L
.1219 .07905 2.37764 .77254 Mabsadd L
.1219 .07905 .56518 .7779 Mabsadd L
.1219 .07905 0 2.5 Mabsadd L closepath F
.12286 .07709 0 2.5 Mabsadd m
.12286 .07709 -0.56518 .7779 Mabsadd L
.12286 .07709 -2.37764 .77254 Mabsadd L
.12286 .07709 -0.91448 -0.29713 Mabsadd L
.12286 .07709 -1.46946 -2.02254 Mabsadd L
.12286 .07709 0 -0.96154 Mabsadd L
.12286 .07709 1.46946 -2.02254 Mabsadd L
.12286 .07709 .91448 -0.29713 Mabsadd L
.12286 .07709 2.37764 .77254 Mabsadd L
.12286 .07709 .56518 .7779 Mabsadd L
.12286 .07709 0 2.5 Mabsadd L closepath F
.12381 .05371 0 2.5 Mabsadd m
.12381 .05371 -0.56518 .7779 Mabsadd L
.12381 .05371 -2.37764 .77254 Mabsadd L
.12381 .05371 -0.91448 -0.29713 Mabsadd L
.12381 .05371 -1.46946 -2.02254 Mabsadd L
.12381 .05371 0 -0.96154 Mabsadd L
.12381 .05371 1.46946 -2.02254 Mabsadd L
.12381 .05371 .91448 -0.29713 Mabsadd L
.12381 .05371 2.37764 .77254 Mabsadd L
.12381 .05371 .56518 .7779 Mabsadd L
.12381 .05371 0 2.5 Mabsadd L closepath F
.12476 .03308 0 2.5 Mabsadd m
.12476 .03308 -0.56518 .7779 Mabsadd L
.12476 .03308 -2.37764 .77254 Mabsadd L
.12476 .03308 -0.91448 -0.29713 Mabsadd L
.12476 .03308 -1.46946 -2.02254 Mabsadd L
.12476 .03308 0 -0.96154 Mabsadd L
.12476 .03308 1.46946 -2.02254 Mabsadd L
.12476 .03308 .91448 -0.29713 Mabsadd L
.12476 .03308 2.37764 .77254 Mabsadd L
.12476 .03308 .56518 .7779 Mabsadd L
.12476 .03308 0 2.5 Mabsadd L closepath F
.12571 .03309 0 2.5 Mabsadd m
.12571 .03309 -0.56518 .7779 Mabsadd L
.12571 .03309 -2.37764 .77254 Mabsadd L
.12571 .03309 -0.91448 -0.29713 Mabsadd L
.12571 .03309 -1.46946 -2.02254 Mabsadd L
.12571 .03309 0 -0.96154 Mabsadd L
.12571 .03309 1.46946 -2.02254 Mabsadd L
.12571 .03309 .91448 -0.29713 Mabsadd L
.12571 .03309 2.37764 .77254 Mabsadd L
.12571 .03309 .56518 .7779 Mabsadd L
.12571 .03309 0 2.5 Mabsadd L closepath F
.12667 .03364 0 2.5 Mabsadd m
.12667 .03364 -0.56518 .7779 Mabsadd L
.12667 .03364 -2.37764 .77254 Mabsadd L
.12667 .03364 -0.91448 -0.29713 Mabsadd L
.12667 .03364 -1.46946 -2.02254 Mabsadd L
.12667 .03364 0 -0.96154 Mabsadd L
.12667 .03364 1.46946 -2.02254 Mabsadd L
.12667 .03364 .91448 -0.29713 Mabsadd L
.12667 .03364 2.37764 .77254 Mabsadd L
.12667 .03364 .56518 .7779 Mabsadd L
.12667 .03364 0 2.5 Mabsadd L closepath F
.12762 .04461 0 2.5 Mabsadd m
.12762 .04461 -0.56518 .7779 Mabsadd L
.12762 .04461 -2.37764 .77254 Mabsadd L
.12762 .04461 -0.91448 -0.29713 Mabsadd L
.12762 .04461 -1.46946 -2.02254 Mabsadd L
.12762 .04461 0 -0.96154 Mabsadd L
.12762 .04461 1.46946 -2.02254 Mabsadd L
.12762 .04461 .91448 -0.29713 Mabsadd L
.12762 .04461 2.37764 .77254 Mabsadd L
.12762 .04461 .56518 .7779 Mabsadd L
.12762 .04461 0 2.5 Mabsadd L closepath F
.12857 .05674 0 2.5 Mabsadd m
.12857 .05674 -0.56518 .7779 Mabsadd L
.12857 .05674 -2.37764 .77254 Mabsadd L
.12857 .05674 -0.91448 -0.29713 Mabsadd L
.12857 .05674 -1.46946 -2.02254 Mabsadd L
.12857 .05674 0 -0.96154 Mabsadd L
.12857 .05674 1.46946 -2.02254 Mabsadd L
.12857 .05674 .91448 -0.29713 Mabsadd L
.12857 .05674 2.37764 .77254 Mabsadd L
.12857 .05674 .56518 .7779 Mabsadd L
.12857 .05674 0 2.5 Mabsadd L closepath F
.12952 .05744 0 2.5 Mabsadd m
.12952 .05744 -0.56518 .7779 Mabsadd L
.12952 .05744 -2.37764 .77254 Mabsadd L
.12952 .05744 -0.91448 -0.29713 Mabsadd L
.12952 .05744 -1.46946 -2.02254 Mabsadd L
.12952 .05744 0 -0.96154 Mabsadd L
.12952 .05744 1.46946 -2.02254 Mabsadd L
.12952 .05744 .91448 -0.29713 Mabsadd L
.12952 .05744 2.37764 .77254 Mabsadd L
.12952 .05744 .56518 .7779 Mabsadd L
.12952 .05744 0 2.5 Mabsadd L closepath F
.13048 .05858 0 2.5 Mabsadd m
.13048 .05858 -0.56518 .7779 Mabsadd L
.13048 .05858 -2.37764 .77254 Mabsadd L
.13048 .05858 -0.91448 -0.29713 Mabsadd L
.13048 .05858 -1.46946 -2.02254 Mabsadd L
.13048 .05858 0 -0.96154 Mabsadd L
.13048 .05858 1.46946 -2.02254 Mabsadd L
.13048 .05858 .91448 -0.29713 Mabsadd L
.13048 .05858 2.37764 .77254 Mabsadd L
.13048 .05858 .56518 .7779 Mabsadd L
.13048 .05858 0 2.5 Mabsadd L closepath F
.13143 .06264 0 2.5 Mabsadd m
.13143 .06264 -0.56518 .7779 Mabsadd L
.13143 .06264 -2.37764 .77254 Mabsadd L
.13143 .06264 -0.91448 -0.29713 Mabsadd L
.13143 .06264 -1.46946 -2.02254 Mabsadd L
.13143 .06264 0 -0.96154 Mabsadd L
.13143 .06264 1.46946 -2.02254 Mabsadd L
.13143 .06264 .91448 -0.29713 Mabsadd L
.13143 .06264 2.37764 .77254 Mabsadd L
.13143 .06264 .56518 .7779 Mabsadd L
.13143 .06264 0 2.5 Mabsadd L closepath F
.13238 .06359 0 2.5 Mabsadd m
.13238 .06359 -0.56518 .7779 Mabsadd L
.13238 .06359 -2.37764 .77254 Mabsadd L
.13238 .06359 -0.91448 -0.29713 Mabsadd L
.13238 .06359 -1.46946 -2.02254 Mabsadd L
.13238 .06359 0 -0.96154 Mabsadd L
.13238 .06359 1.46946 -2.02254 Mabsadd L
.13238 .06359 .91448 -0.29713 Mabsadd L
.13238 .06359 2.37764 .77254 Mabsadd L
.13238 .06359 .56518 .7779 Mabsadd L
.13238 .06359 0 2.5 Mabsadd L closepath F
.13333 .06337 0 2.5 Mabsadd m
.13333 .06337 -0.56518 .7779 Mabsadd L
.13333 .06337 -2.37764 .77254 Mabsadd L
.13333 .06337 -0.91448 -0.29713 Mabsadd L
.13333 .06337 -1.46946 -2.02254 Mabsadd L
.13333 .06337 0 -0.96154 Mabsadd L
.13333 .06337 1.46946 -2.02254 Mabsadd L
.13333 .06337 .91448 -0.29713 Mabsadd L
.13333 .06337 2.37764 .77254 Mabsadd L
.13333 .06337 .56518 .7779 Mabsadd L
.13333 .06337 0 2.5 Mabsadd L closepath F
.13429 .05996 0 2.5 Mabsadd m
.13429 .05996 -0.56518 .7779 Mabsadd L
.13429 .05996 -2.37764 .77254 Mabsadd L
.13429 .05996 -0.91448 -0.29713 Mabsadd L
.13429 .05996 -1.46946 -2.02254 Mabsadd L
.13429 .05996 0 -0.96154 Mabsadd L
.13429 .05996 1.46946 -2.02254 Mabsadd L
.13429 .05996 .91448 -0.29713 Mabsadd L
.13429 .05996 2.37764 .77254 Mabsadd L
.13429 .05996 .56518 .7779 Mabsadd L
.13429 .05996 0 2.5 Mabsadd L closepath F
.13524 .06346 0 2.5 Mabsadd m
.13524 .06346 -0.56518 .7779 Mabsadd L
.13524 .06346 -2.37764 .77254 Mabsadd L
.13524 .06346 -0.91448 -0.29713 Mabsadd L
.13524 .06346 -1.46946 -2.02254 Mabsadd L
.13524 .06346 0 -0.96154 Mabsadd L
.13524 .06346 1.46946 -2.02254 Mabsadd L
.13524 .06346 .91448 -0.29713 Mabsadd L
.13524 .06346 2.37764 .77254 Mabsadd L
.13524 .06346 .56518 .7779 Mabsadd L
.13524 .06346 0 2.5 Mabsadd L closepath F
.13619 .07427 0 2.5 Mabsadd m
.13619 .07427 -0.56518 .7779 Mabsadd L
.13619 .07427 -2.37764 .77254 Mabsadd L
.13619 .07427 -0.91448 -0.29713 Mabsadd L
.13619 .07427 -1.46946 -2.02254 Mabsadd L
.13619 .07427 0 -0.96154 Mabsadd L
.13619 .07427 1.46946 -2.02254 Mabsadd L
.13619 .07427 .91448 -0.29713 Mabsadd L
.13619 .07427 2.37764 .77254 Mabsadd L
.13619 .07427 .56518 .7779 Mabsadd L
.13619 .07427 0 2.5 Mabsadd L closepath F
.13714 .0743 0 2.5 Mabsadd m
.13714 .0743 -0.56518 .7779 Mabsadd L
.13714 .0743 -2.37764 .77254 Mabsadd L
.13714 .0743 -0.91448 -0.29713 Mabsadd L
.13714 .0743 -1.46946 -2.02254 Mabsadd L
.13714 .0743 0 -0.96154 Mabsadd L
.13714 .0743 1.46946 -2.02254 Mabsadd L
.13714 .0743 .91448 -0.29713 Mabsadd L
.13714 .0743 2.37764 .77254 Mabsadd L
.13714 .0743 .56518 .7779 Mabsadd L
.13714 .0743 0 2.5 Mabsadd L closepath F
.1381 .07437 0 2.5 Mabsadd m
.1381 .07437 -0.56518 .7779 Mabsadd L
.1381 .07437 -2.37764 .77254 Mabsadd L
.1381 .07437 -0.91448 -0.29713 Mabsadd L
.1381 .07437 -1.46946 -2.02254 Mabsadd L
.1381 .07437 0 -0.96154 Mabsadd L
.1381 .07437 1.46946 -2.02254 Mabsadd L
.1381 .07437 .91448 -0.29713 Mabsadd L
.1381 .07437 2.37764 .77254 Mabsadd L
.1381 .07437 .56518 .7779 Mabsadd L
.1381 .07437 0 2.5 Mabsadd L closepath F
.13905 .0744 0 2.5 Mabsadd m
.13905 .0744 -0.56518 .7779 Mabsadd L
.13905 .0744 -2.37764 .77254 Mabsadd L
.13905 .0744 -0.91448 -0.29713 Mabsadd L
.13905 .0744 -1.46946 -2.02254 Mabsadd L
.13905 .0744 0 -0.96154 Mabsadd L
.13905 .0744 1.46946 -2.02254 Mabsadd L
.13905 .0744 .91448 -0.29713 Mabsadd L
.13905 .0744 2.37764 .77254 Mabsadd L
.13905 .0744 .56518 .7779 Mabsadd L
.13905 .0744 0 2.5 Mabsadd L closepath F
.14 .07436 0 2.5 Mabsadd m
.14 .07436 -0.56518 .7779 Mabsadd L
.14 .07436 -2.37764 .77254 Mabsadd L
.14 .07436 -0.91448 -0.29713 Mabsadd L
.14 .07436 -1.46946 -2.02254 Mabsadd L
.14 .07436 0 -0.96154 Mabsadd L
.14 .07436 1.46946 -2.02254 Mabsadd L
.14 .07436 .91448 -0.29713 Mabsadd L
.14 .07436 2.37764 .77254 Mabsadd L
.14 .07436 .56518 .7779 Mabsadd L
.14 .07436 0 2.5 Mabsadd L closepath F
.14095 .07428 0 2.5 Mabsadd m
.14095 .07428 -0.56518 .7779 Mabsadd L
.14095 .07428 -2.37764 .77254 Mabsadd L
.14095 .07428 -0.91448 -0.29713 Mabsadd L
.14095 .07428 -1.46946 -2.02254 Mabsadd L
.14095 .07428 0 -0.96154 Mabsadd L
.14095 .07428 1.46946 -2.02254 Mabsadd L
.14095 .07428 .91448 -0.29713 Mabsadd L
.14095 .07428 2.37764 .77254 Mabsadd L
.14095 .07428 .56518 .7779 Mabsadd L
.14095 .07428 0 2.5 Mabsadd L closepath F
.1419 .07472 0 2.5 Mabsadd m
.1419 .07472 -0.56518 .7779 Mabsadd L
.1419 .07472 -2.37764 .77254 Mabsadd L
.1419 .07472 -0.91448 -0.29713 Mabsadd L
.1419 .07472 -1.46946 -2.02254 Mabsadd L
.1419 .07472 0 -0.96154 Mabsadd L
.1419 .07472 1.46946 -2.02254 Mabsadd L
.1419 .07472 .91448 -0.29713 Mabsadd L
.1419 .07472 2.37764 .77254 Mabsadd L
.1419 .07472 .56518 .7779 Mabsadd L
.1419 .07472 0 2.5 Mabsadd L closepath F
.14286 .07611 0 2.5 Mabsadd m
.14286 .07611 -0.56518 .7779 Mabsadd L
.14286 .07611 -2.37764 .77254 Mabsadd L
.14286 .07611 -0.91448 -0.29713 Mabsadd L
.14286 .07611 -1.46946 -2.02254 Mabsadd L
.14286 .07611 0 -0.96154 Mabsadd L
.14286 .07611 1.46946 -2.02254 Mabsadd L
.14286 .07611 .91448 -0.29713 Mabsadd L
.14286 .07611 2.37764 .77254 Mabsadd L
.14286 .07611 .56518 .7779 Mabsadd L
.14286 .07611 0 2.5 Mabsadd L closepath F
.14381 .07619 0 2.5 Mabsadd m
.14381 .07619 -0.56518 .7779 Mabsadd L
.14381 .07619 -2.37764 .77254 Mabsadd L
.14381 .07619 -0.91448 -0.29713 Mabsadd L
.14381 .07619 -1.46946 -2.02254 Mabsadd L
.14381 .07619 0 -0.96154 Mabsadd L
.14381 .07619 1.46946 -2.02254 Mabsadd L
.14381 .07619 .91448 -0.29713 Mabsadd L
.14381 .07619 2.37764 .77254 Mabsadd L
.14381 .07619 .56518 .7779 Mabsadd L
.14381 .07619 0 2.5 Mabsadd L closepath F
.14476 .07646 0 2.5 Mabsadd m
.14476 .07646 -0.56518 .7779 Mabsadd L
.14476 .07646 -2.37764 .77254 Mabsadd L
.14476 .07646 -0.91448 -0.29713 Mabsadd L
.14476 .07646 -1.46946 -2.02254 Mabsadd L
.14476 .07646 0 -0.96154 Mabsadd L
.14476 .07646 1.46946 -2.02254 Mabsadd L
.14476 .07646 .91448 -0.29713 Mabsadd L
.14476 .07646 2.37764 .77254 Mabsadd L
.14476 .07646 .56518 .7779 Mabsadd L
.14476 .07646 0 2.5 Mabsadd L closepath F
.14571 .07648 0 2.5 Mabsadd m
.14571 .07648 -0.56518 .7779 Mabsadd L
.14571 .07648 -2.37764 .77254 Mabsadd L
.14571 .07648 -0.91448 -0.29713 Mabsadd L
.14571 .07648 -1.46946 -2.02254 Mabsadd L
.14571 .07648 0 -0.96154 Mabsadd L
.14571 .07648 1.46946 -2.02254 Mabsadd L
.14571 .07648 .91448 -0.29713 Mabsadd L
.14571 .07648 2.37764 .77254 Mabsadd L
.14571 .07648 .56518 .7779 Mabsadd L
.14571 .07648 0 2.5 Mabsadd L closepath F
.14667 .07654 0 2.5 Mabsadd m
.14667 .07654 -0.56518 .7779 Mabsadd L
.14667 .07654 -2.37764 .77254 Mabsadd L
.14667 .07654 -0.91448 -0.29713 Mabsadd L
.14667 .07654 -1.46946 -2.02254 Mabsadd L
.14667 .07654 0 -0.96154 Mabsadd L
.14667 .07654 1.46946 -2.02254 Mabsadd L
.14667 .07654 .91448 -0.29713 Mabsadd L
.14667 .07654 2.37764 .77254 Mabsadd L
.14667 .07654 .56518 .7779 Mabsadd L
.14667 .07654 0 2.5 Mabsadd L closepath F
.14762 .07665 0 2.5 Mabsadd m
.14762 .07665 -0.56518 .7779 Mabsadd L
.14762 .07665 -2.37764 .77254 Mabsadd L
.14762 .07665 -0.91448 -0.29713 Mabsadd L
.14762 .07665 -1.46946 -2.02254 Mabsadd L
.14762 .07665 0 -0.96154 Mabsadd L
.14762 .07665 1.46946 -2.02254 Mabsadd L
.14762 .07665 .91448 -0.29713 Mabsadd L
.14762 .07665 2.37764 .77254 Mabsadd L
.14762 .07665 .56518 .7779 Mabsadd L
.14762 .07665 0 2.5 Mabsadd L closepath F
.14857 .07665 0 2.5 Mabsadd m
.14857 .07665 -0.56518 .7779 Mabsadd L
.14857 .07665 -2.37764 .77254 Mabsadd L
.14857 .07665 -0.91448 -0.29713 Mabsadd L
.14857 .07665 -1.46946 -2.02254 Mabsadd L
.14857 .07665 0 -0.96154 Mabsadd L
.14857 .07665 1.46946 -2.02254 Mabsadd L
.14857 .07665 .91448 -0.29713 Mabsadd L
.14857 .07665 2.37764 .77254 Mabsadd L
.14857 .07665 .56518 .7779 Mabsadd L
.14857 .07665 0 2.5 Mabsadd L closepath F
.14952 .07666 0 2.5 Mabsadd m
.14952 .07666 -0.56518 .7779 Mabsadd L
.14952 .07666 -2.37764 .77254 Mabsadd L
.14952 .07666 -0.91448 -0.29713 Mabsadd L
.14952 .07666 -1.46946 -2.02254 Mabsadd L
.14952 .07666 0 -0.96154 Mabsadd L
.14952 .07666 1.46946 -2.02254 Mabsadd L
.14952 .07666 .91448 -0.29713 Mabsadd L
.14952 .07666 2.37764 .77254 Mabsadd L
.14952 .07666 .56518 .7779 Mabsadd L
.14952 .07666 0 2.5 Mabsadd L closepath F
.15048 .07668 0 2.5 Mabsadd m
.15048 .07668 -0.56518 .7779 Mabsadd L
.15048 .07668 -2.37764 .77254 Mabsadd L
.15048 .07668 -0.91448 -0.29713 Mabsadd L
.15048 .07668 -1.46946 -2.02254 Mabsadd L
.15048 .07668 0 -0.96154 Mabsadd L
.15048 .07668 1.46946 -2.02254 Mabsadd L
.15048 .07668 .91448 -0.29713 Mabsadd L
.15048 .07668 2.37764 .77254 Mabsadd L
.15048 .07668 .56518 .7779 Mabsadd L
.15048 .07668 0 2.5 Mabsadd L closepath F
.15143 .07669 0 2.5 Mabsadd m
.15143 .07669 -0.56518 .7779 Mabsadd L
.15143 .07669 -2.37764 .77254 Mabsadd L
.15143 .07669 -0.91448 -0.29713 Mabsadd L
.15143 .07669 -1.46946 -2.02254 Mabsadd L
.15143 .07669 0 -0.96154 Mabsadd L
.15143 .07669 1.46946 -2.02254 Mabsadd L
.15143 .07669 .91448 -0.29713 Mabsadd L
.15143 .07669 2.37764 .77254 Mabsadd L
.15143 .07669 .56518 .7779 Mabsadd L
.15143 .07669 0 2.5 Mabsadd L closepath F
.15238 .07669 0 2.5 Mabsadd m
.15238 .07669 -0.56518 .7779 Mabsadd L
.15238 .07669 -2.37764 .77254 Mabsadd L
.15238 .07669 -0.91448 -0.29713 Mabsadd L
.15238 .07669 -1.46946 -2.02254 Mabsadd L
.15238 .07669 0 -0.96154 Mabsadd L
.15238 .07669 1.46946 -2.02254 Mabsadd L
.15238 .07669 .91448 -0.29713 Mabsadd L
.15238 .07669 2.37764 .77254 Mabsadd L
.15238 .07669 .56518 .7779 Mabsadd L
.15238 .07669 0 2.5 Mabsadd L closepath F
.15333 .07669 0 2.5 Mabsadd m
.15333 .07669 -0.56518 .7779 Mabsadd L
.15333 .07669 -2.37764 .77254 Mabsadd L
.15333 .07669 -0.91448 -0.29713 Mabsadd L
.15333 .07669 -1.46946 -2.02254 Mabsadd L
.15333 .07669 0 -0.96154 Mabsadd L
.15333 .07669 1.46946 -2.02254 Mabsadd L
.15333 .07669 .91448 -0.29713 Mabsadd L
.15333 .07669 2.37764 .77254 Mabsadd L
.15333 .07669 .56518 .7779 Mabsadd L
.15333 .07669 0 2.5 Mabsadd L closepath F
.15429 .0767 0 2.5 Mabsadd m
.15429 .0767 -0.56518 .7779 Mabsadd L
.15429 .0767 -2.37764 .77254 Mabsadd L
.15429 .0767 -0.91448 -0.29713 Mabsadd L
.15429 .0767 -1.46946 -2.02254 Mabsadd L
.15429 .0767 0 -0.96154 Mabsadd L
.15429 .0767 1.46946 -2.02254 Mabsadd L
.15429 .0767 .91448 -0.29713 Mabsadd L
.15429 .0767 2.37764 .77254 Mabsadd L
.15429 .0767 .56518 .7779 Mabsadd L
.15429 .0767 0 2.5 Mabsadd L closepath F
.15524 .0767 0 2.5 Mabsadd m
.15524 .0767 -0.56518 .7779 Mabsadd L
.15524 .0767 -2.37764 .77254 Mabsadd L
.15524 .0767 -0.91448 -0.29713 Mabsadd L
.15524 .0767 -1.46946 -2.02254 Mabsadd L
.15524 .0767 0 -0.96154 Mabsadd L
.15524 .0767 1.46946 -2.02254 Mabsadd L
.15524 .0767 .91448 -0.29713 Mabsadd L
.15524 .0767 2.37764 .77254 Mabsadd L
.15524 .0767 .56518 .7779 Mabsadd L
.15524 .0767 0 2.5 Mabsadd L closepath F
.15619 .0767 0 2.5 Mabsadd m
.15619 .0767 -0.56518 .7779 Mabsadd L
.15619 .0767 -2.37764 .77254 Mabsadd L
.15619 .0767 -0.91448 -0.29713 Mabsadd L
.15619 .0767 -1.46946 -2.02254 Mabsadd L
.15619 .0767 0 -0.96154 Mabsadd L
.15619 .0767 1.46946 -2.02254 Mabsadd L
.15619 .0767 .91448 -0.29713 Mabsadd L
.15619 .0767 2.37764 .77254 Mabsadd L
.15619 .0767 .56518 .7779 Mabsadd L
.15619 .0767 0 2.5 Mabsadd L closepath F
.15714 .0767 0 2.5 Mabsadd m
.15714 .0767 -0.56518 .7779 Mabsadd L
.15714 .0767 -2.37764 .77254 Mabsadd L
.15714 .0767 -0.91448 -0.29713 Mabsadd L
.15714 .0767 -1.46946 -2.02254 Mabsadd L
.15714 .0767 0 -0.96154 Mabsadd L
.15714 .0767 1.46946 -2.02254 Mabsadd L
.15714 .0767 .91448 -0.29713 Mabsadd L
.15714 .0767 2.37764 .77254 Mabsadd L
.15714 .0767 .56518 .7779 Mabsadd L
.15714 .0767 0 2.5 Mabsadd L closepath F
.1581 .0767 0 2.5 Mabsadd m
.1581 .0767 -0.56518 .7779 Mabsadd L
.1581 .0767 -2.37764 .77254 Mabsadd L
.1581 .0767 -0.91448 -0.29713 Mabsadd L
.1581 .0767 -1.46946 -2.02254 Mabsadd L
.1581 .0767 0 -0.96154 Mabsadd L
.1581 .0767 1.46946 -2.02254 Mabsadd L
.1581 .0767 .91448 -0.29713 Mabsadd L
.1581 .0767 2.37764 .77254 Mabsadd L
.1581 .0767 .56518 .7779 Mabsadd L
.1581 .0767 0 2.5 Mabsadd L closepath F
.15905 .0767 0 2.5 Mabsadd m
.15905 .0767 -0.56518 .7779 Mabsadd L
.15905 .0767 -2.37764 .77254 Mabsadd L
.15905 .0767 -0.91448 -0.29713 Mabsadd L
.15905 .0767 -1.46946 -2.02254 Mabsadd L
.15905 .0767 0 -0.96154 Mabsadd L
.15905 .0767 1.46946 -2.02254 Mabsadd L
.15905 .0767 .91448 -0.29713 Mabsadd L
.15905 .0767 2.37764 .77254 Mabsadd L
.15905 .0767 .56518 .7779 Mabsadd L
.15905 .0767 0 2.5 Mabsadd L closepath F
.16 .0767 0 2.5 Mabsadd m
.16 .0767 -0.56518 .7779 Mabsadd L
.16 .0767 -2.37764 .77254 Mabsadd L
.16 .0767 -0.91448 -0.29713 Mabsadd L
.16 .0767 -1.46946 -2.02254 Mabsadd L
.16 .0767 0 -0.96154 Mabsadd L
.16 .0767 1.46946 -2.02254 Mabsadd L
.16 .0767 .91448 -0.29713 Mabsadd L
.16 .0767 2.37764 .77254 Mabsadd L
.16 .0767 .56518 .7779 Mabsadd L
.16 .0767 0 2.5 Mabsadd L closepath F
.16095 .07669 0 2.5 Mabsadd m
.16095 .07669 -0.56518 .7779 Mabsadd L
.16095 .07669 -2.37764 .77254 Mabsadd L
.16095 .07669 -0.91448 -0.29713 Mabsadd L
.16095 .07669 -1.46946 -2.02254 Mabsadd L
.16095 .07669 0 -0.96154 Mabsadd L
.16095 .07669 1.46946 -2.02254 Mabsadd L
.16095 .07669 .91448 -0.29713 Mabsadd L
.16095 .07669 2.37764 .77254 Mabsadd L
.16095 .07669 .56518 .7779 Mabsadd L
.16095 .07669 0 2.5 Mabsadd L closepath F
.1619 .07668 0 2.5 Mabsadd m
.1619 .07668 -0.56518 .7779 Mabsadd L
.1619 .07668 -2.37764 .77254 Mabsadd L
.1619 .07668 -0.91448 -0.29713 Mabsadd L
.1619 .07668 -1.46946 -2.02254 Mabsadd L
.1619 .07668 0 -0.96154 Mabsadd L
.1619 .07668 1.46946 -2.02254 Mabsadd L
.1619 .07668 .91448 -0.29713 Mabsadd L
.1619 .07668 2.37764 .77254 Mabsadd L
.1619 .07668 .56518 .7779 Mabsadd L
.1619 .07668 0 2.5 Mabsadd L closepath F
.16286 .07668 0 2.5 Mabsadd m
.16286 .07668 -0.56518 .7779 Mabsadd L
.16286 .07668 -2.37764 .77254 Mabsadd L
.16286 .07668 -0.91448 -0.29713 Mabsadd L
.16286 .07668 -1.46946 -2.02254 Mabsadd L
.16286 .07668 0 -0.96154 Mabsadd L
.16286 .07668 1.46946 -2.02254 Mabsadd L
.16286 .07668 .91448 -0.29713 Mabsadd L
.16286 .07668 2.37764 .77254 Mabsadd L
.16286 .07668 .56518 .7779 Mabsadd L
.16286 .07668 0 2.5 Mabsadd L closepath F
.16381 .07668 0 2.5 Mabsadd m
.16381 .07668 -0.56518 .7779 Mabsadd L
.16381 .07668 -2.37764 .77254 Mabsadd L
.16381 .07668 -0.91448 -0.29713 Mabsadd L
.16381 .07668 -1.46946 -2.02254 Mabsadd L
.16381 .07668 0 -0.96154 Mabsadd L
.16381 .07668 1.46946 -2.02254 Mabsadd L
.16381 .07668 .91448 -0.29713 Mabsadd L
.16381 .07668 2.37764 .77254 Mabsadd L
.16381 .07668 .56518 .7779 Mabsadd L
.16381 .07668 0 2.5 Mabsadd L closepath F
.16476 .07668 0 2.5 Mabsadd m
.16476 .07668 -0.56518 .7779 Mabsadd L
.16476 .07668 -2.37764 .77254 Mabsadd L
.16476 .07668 -0.91448 -0.29713 Mabsadd L
.16476 .07668 -1.46946 -2.02254 Mabsadd L
.16476 .07668 0 -0.96154 Mabsadd L
.16476 .07668 1.46946 -2.02254 Mabsadd L
.16476 .07668 .91448 -0.29713 Mabsadd L
.16476 .07668 2.37764 .77254 Mabsadd L
.16476 .07668 .56518 .7779 Mabsadd L
.16476 .07668 0 2.5 Mabsadd L closepath F
.16571 .07666 0 2.5 Mabsadd m
.16571 .07666 -0.56518 .7779 Mabsadd L
.16571 .07666 -2.37764 .77254 Mabsadd L
.16571 .07666 -0.91448 -0.29713 Mabsadd L
.16571 .07666 -1.46946 -2.02254 Mabsadd L
.16571 .07666 0 -0.96154 Mabsadd L
.16571 .07666 1.46946 -2.02254 Mabsadd L
.16571 .07666 .91448 -0.29713 Mabsadd L
.16571 .07666 2.37764 .77254 Mabsadd L
.16571 .07666 .56518 .7779 Mabsadd L
.16571 .07666 0 2.5 Mabsadd L closepath F
.16667 .07666 0 2.5 Mabsadd m
.16667 .07666 -0.56518 .7779 Mabsadd L
.16667 .07666 -2.37764 .77254 Mabsadd L
.16667 .07666 -0.91448 -0.29713 Mabsadd L
.16667 .07666 -1.46946 -2.02254 Mabsadd L
.16667 .07666 0 -0.96154 Mabsadd L
.16667 .07666 1.46946 -2.02254 Mabsadd L
.16667 .07666 .91448 -0.29713 Mabsadd L
.16667 .07666 2.37764 .77254 Mabsadd L
.16667 .07666 .56518 .7779 Mabsadd L
.16667 .07666 0 2.5 Mabsadd L closepath F
.16762 .07665 0 2.5 Mabsadd m
.16762 .07665 -0.56518 .7779 Mabsadd L
.16762 .07665 -2.37764 .77254 Mabsadd L
.16762 .07665 -0.91448 -0.29713 Mabsadd L
.16762 .07665 -1.46946 -2.02254 Mabsadd L
.16762 .07665 0 -0.96154 Mabsadd L
.16762 .07665 1.46946 -2.02254 Mabsadd L
.16762 .07665 .91448 -0.29713 Mabsadd L
.16762 .07665 2.37764 .77254 Mabsadd L
.16762 .07665 .56518 .7779 Mabsadd L
.16762 .07665 0 2.5 Mabsadd L closepath F
.16857 .07661 0 2.5 Mabsadd m
.16857 .07661 -0.56518 .7779 Mabsadd L
.16857 .07661 -2.37764 .77254 Mabsadd L
.16857 .07661 -0.91448 -0.29713 Mabsadd L
.16857 .07661 -1.46946 -2.02254 Mabsadd L
.16857 .07661 0 -0.96154 Mabsadd L
.16857 .07661 1.46946 -2.02254 Mabsadd L
.16857 .07661 .91448 -0.29713 Mabsadd L
.16857 .07661 2.37764 .77254 Mabsadd L
.16857 .07661 .56518 .7779 Mabsadd L
.16857 .07661 0 2.5 Mabsadd L closepath F
.16952 .07661 0 2.5 Mabsadd m
.16952 .07661 -0.56518 .7779 Mabsadd L
.16952 .07661 -2.37764 .77254 Mabsadd L
.16952 .07661 -0.91448 -0.29713 Mabsadd L
.16952 .07661 -1.46946 -2.02254 Mabsadd L
.16952 .07661 0 -0.96154 Mabsadd L
.16952 .07661 1.46946 -2.02254 Mabsadd L
.16952 .07661 .91448 -0.29713 Mabsadd L
.16952 .07661 2.37764 .77254 Mabsadd L
.16952 .07661 .56518 .7779 Mabsadd L
.16952 .07661 0 2.5 Mabsadd L closepath F
.17048 .07655 0 2.5 Mabsadd m
.17048 .07655 -0.56518 .7779 Mabsadd L
.17048 .07655 -2.37764 .77254 Mabsadd L
.17048 .07655 -0.91448 -0.29713 Mabsadd L
.17048 .07655 -1.46946 -2.02254 Mabsadd L
.17048 .07655 0 -0.96154 Mabsadd L
.17048 .07655 1.46946 -2.02254 Mabsadd L
.17048 .07655 .91448 -0.29713 Mabsadd L
.17048 .07655 2.37764 .77254 Mabsadd L
.17048 .07655 .56518 .7779 Mabsadd L
.17048 .07655 0 2.5 Mabsadd L closepath F
.17143 .07653 0 2.5 Mabsadd m
.17143 .07653 -0.56518 .7779 Mabsadd L
.17143 .07653 -2.37764 .77254 Mabsadd L
.17143 .07653 -0.91448 -0.29713 Mabsadd L
.17143 .07653 -1.46946 -2.02254 Mabsadd L
.17143 .07653 0 -0.96154 Mabsadd L
.17143 .07653 1.46946 -2.02254 Mabsadd L
.17143 .07653 .91448 -0.29713 Mabsadd L
.17143 .07653 2.37764 .77254 Mabsadd L
.17143 .07653 .56518 .7779 Mabsadd L
.17143 .07653 0 2.5 Mabsadd L closepath F
.17238 .07632 0 2.5 Mabsadd m
.17238 .07632 -0.56518 .7779 Mabsadd L
.17238 .07632 -2.37764 .77254 Mabsadd L
.17238 .07632 -0.91448 -0.29713 Mabsadd L
.17238 .07632 -1.46946 -2.02254 Mabsadd L
.17238 .07632 0 -0.96154 Mabsadd L
.17238 .07632 1.46946 -2.02254 Mabsadd L
.17238 .07632 .91448 -0.29713 Mabsadd L
.17238 .07632 2.37764 .77254 Mabsadd L
.17238 .07632 .56518 .7779 Mabsadd L
.17238 .07632 0 2.5 Mabsadd L closepath F
.17333 .07622 0 2.5 Mabsadd m
.17333 .07622 -0.56518 .7779 Mabsadd L
.17333 .07622 -2.37764 .77254 Mabsadd L
.17333 .07622 -0.91448 -0.29713 Mabsadd L
.17333 .07622 -1.46946 -2.02254 Mabsadd L
.17333 .07622 0 -0.96154 Mabsadd L
.17333 .07622 1.46946 -2.02254 Mabsadd L
.17333 .07622 .91448 -0.29713 Mabsadd L
.17333 .07622 2.37764 .77254 Mabsadd L
.17333 .07622 .56518 .7779 Mabsadd L
.17333 .07622 0 2.5 Mabsadd L closepath F
.17429 .07518 0 2.5 Mabsadd m
.17429 .07518 -0.56518 .7779 Mabsadd L
.17429 .07518 -2.37764 .77254 Mabsadd L
.17429 .07518 -0.91448 -0.29713 Mabsadd L
.17429 .07518 -1.46946 -2.02254 Mabsadd L
.17429 .07518 0 -0.96154 Mabsadd L
.17429 .07518 1.46946 -2.02254 Mabsadd L
.17429 .07518 .91448 -0.29713 Mabsadd L
.17429 .07518 2.37764 .77254 Mabsadd L
.17429 .07518 .56518 .7779 Mabsadd L
.17429 .07518 0 2.5 Mabsadd L closepath F
.17524 .07456 0 2.5 Mabsadd m
.17524 .07456 -0.56518 .7779 Mabsadd L
.17524 .07456 -2.37764 .77254 Mabsadd L
.17524 .07456 -0.91448 -0.29713 Mabsadd L
.17524 .07456 -1.46946 -2.02254 Mabsadd L
.17524 .07456 0 -0.96154 Mabsadd L
.17524 .07456 1.46946 -2.02254 Mabsadd L
.17524 .07456 .91448 -0.29713 Mabsadd L
.17524 .07456 2.37764 .77254 Mabsadd L
.17524 .07456 .56518 .7779 Mabsadd L
.17524 .07456 0 2.5 Mabsadd L closepath F
.17619 .06767 0 2.5 Mabsadd m
.17619 .06767 -0.56518 .7779 Mabsadd L
.17619 .06767 -2.37764 .77254 Mabsadd L
.17619 .06767 -0.91448 -0.29713 Mabsadd L
.17619 .06767 -1.46946 -2.02254 Mabsadd L
.17619 .06767 0 -0.96154 Mabsadd L
.17619 .06767 1.46946 -2.02254 Mabsadd L
.17619 .06767 .91448 -0.29713 Mabsadd L
.17619 .06767 2.37764 .77254 Mabsadd L
.17619 .06767 .56518 .7779 Mabsadd L
.17619 .06767 0 2.5 Mabsadd L closepath F
.17714 .06247 0 2.5 Mabsadd m
.17714 .06247 -0.56518 .7779 Mabsadd L
.17714 .06247 -2.37764 .77254 Mabsadd L
.17714 .06247 -0.91448 -0.29713 Mabsadd L
.17714 .06247 -1.46946 -2.02254 Mabsadd L
.17714 .06247 0 -0.96154 Mabsadd L
.17714 .06247 1.46946 -2.02254 Mabsadd L
.17714 .06247 .91448 -0.29713 Mabsadd L
.17714 .06247 2.37764 .77254 Mabsadd L
.17714 .06247 .56518 .7779 Mabsadd L
.17714 .06247 0 2.5 Mabsadd L closepath F
.1781 .04488 0 2.5 Mabsadd m
.1781 .04488 -0.56518 .7779 Mabsadd L
.1781 .04488 -2.37764 .77254 Mabsadd L
.1781 .04488 -0.91448 -0.29713 Mabsadd L
.1781 .04488 -1.46946 -2.02254 Mabsadd L
.1781 .04488 0 -0.96154 Mabsadd L
.1781 .04488 1.46946 -2.02254 Mabsadd L
.1781 .04488 .91448 -0.29713 Mabsadd L
.1781 .04488 2.37764 .77254 Mabsadd L
.1781 .04488 .56518 .7779 Mabsadd L
.1781 .04488 0 2.5 Mabsadd L closepath F
.17905 .04488 0 2.5 Mabsadd m
.17905 .04488 -0.56518 .7779 Mabsadd L
.17905 .04488 -2.37764 .77254 Mabsadd L
.17905 .04488 -0.91448 -0.29713 Mabsadd L
.17905 .04488 -1.46946 -2.02254 Mabsadd L
.17905 .04488 0 -0.96154 Mabsadd L
.17905 .04488 1.46946 -2.02254 Mabsadd L
.17905 .04488 .91448 -0.29713 Mabsadd L
.17905 .04488 2.37764 .77254 Mabsadd L
.17905 .04488 .56518 .7779 Mabsadd L
.17905 .04488 0 2.5 Mabsadd L closepath F
.18 .04529 0 2.5 Mabsadd m
.18 .04529 -0.56518 .7779 Mabsadd L
.18 .04529 -2.37764 .77254 Mabsadd L
.18 .04529 -0.91448 -0.29713 Mabsadd L
.18 .04529 -1.46946 -2.02254 Mabsadd L
.18 .04529 0 -0.96154 Mabsadd L
.18 .04529 1.46946 -2.02254 Mabsadd L
.18 .04529 .91448 -0.29713 Mabsadd L
.18 .04529 2.37764 .77254 Mabsadd L
.18 .04529 .56518 .7779 Mabsadd L
.18 .04529 0 2.5 Mabsadd L closepath F
.18095 .0535 0 2.5 Mabsadd m
.18095 .0535 -0.56518 .7779 Mabsadd L
.18095 .0535 -2.37764 .77254 Mabsadd L
.18095 .0535 -0.91448 -0.29713 Mabsadd L
.18095 .0535 -1.46946 -2.02254 Mabsadd L
.18095 .0535 0 -0.96154 Mabsadd L
.18095 .0535 1.46946 -2.02254 Mabsadd L
.18095 .0535 .91448 -0.29713 Mabsadd L
.18095 .0535 2.37764 .77254 Mabsadd L
.18095 .0535 .56518 .7779 Mabsadd L
.18095 .0535 0 2.5 Mabsadd L closepath F
.1819 .06214 0 2.5 Mabsadd m
.1819 .06214 -0.56518 .7779 Mabsadd L
.1819 .06214 -2.37764 .77254 Mabsadd L
.1819 .06214 -0.91448 -0.29713 Mabsadd L
.1819 .06214 -1.46946 -2.02254 Mabsadd L
.1819 .06214 0 -0.96154 Mabsadd L
.1819 .06214 1.46946 -2.02254 Mabsadd L
.1819 .06214 .91448 -0.29713 Mabsadd L
.1819 .06214 2.37764 .77254 Mabsadd L
.1819 .06214 .56518 .7779 Mabsadd L
.1819 .06214 0 2.5 Mabsadd L closepath F
.18286 .06259 0 2.5 Mabsadd m
.18286 .06259 -0.56518 .7779 Mabsadd L
.18286 .06259 -2.37764 .77254 Mabsadd L
.18286 .06259 -0.91448 -0.29713 Mabsadd L
.18286 .06259 -1.46946 -2.02254 Mabsadd L
.18286 .06259 0 -0.96154 Mabsadd L
.18286 .06259 1.46946 -2.02254 Mabsadd L
.18286 .06259 .91448 -0.29713 Mabsadd L
.18286 .06259 2.37764 .77254 Mabsadd L
.18286 .06259 .56518 .7779 Mabsadd L
.18286 .06259 0 2.5 Mabsadd L closepath F
.18381 .06344 0 2.5 Mabsadd m
.18381 .06344 -0.56518 .7779 Mabsadd L
.18381 .06344 -2.37764 .77254 Mabsadd L
.18381 .06344 -0.91448 -0.29713 Mabsadd L
.18381 .06344 -1.46946 -2.02254 Mabsadd L
.18381 .06344 0 -0.96154 Mabsadd L
.18381 .06344 1.46946 -2.02254 Mabsadd L
.18381 .06344 .91448 -0.29713 Mabsadd L
.18381 .06344 2.37764 .77254 Mabsadd L
.18381 .06344 .56518 .7779 Mabsadd L
.18381 .06344 0 2.5 Mabsadd L closepath F
.18476 .06731 0 2.5 Mabsadd m
.18476 .06731 -0.56518 .7779 Mabsadd L
.18476 .06731 -2.37764 .77254 Mabsadd L
.18476 .06731 -0.91448 -0.29713 Mabsadd L
.18476 .06731 -1.46946 -2.02254 Mabsadd L
.18476 .06731 0 -0.96154 Mabsadd L
.18476 .06731 1.46946 -2.02254 Mabsadd L
.18476 .06731 .91448 -0.29713 Mabsadd L
.18476 .06731 2.37764 .77254 Mabsadd L
.18476 .06731 .56518 .7779 Mabsadd L
.18476 .06731 0 2.5 Mabsadd L closepath F
.18571 .0688 0 2.5 Mabsadd m
.18571 .0688 -0.56518 .7779 Mabsadd L
.18571 .0688 -2.37764 .77254 Mabsadd L
.18571 .0688 -0.91448 -0.29713 Mabsadd L
.18571 .0688 -1.46946 -2.02254 Mabsadd L
.18571 .0688 0 -0.96154 Mabsadd L
.18571 .0688 1.46946 -2.02254 Mabsadd L
.18571 .0688 .91448 -0.29713 Mabsadd L
.18571 .0688 2.37764 .77254 Mabsadd L
.18571 .0688 .56518 .7779 Mabsadd L
.18571 .0688 0 2.5 Mabsadd L closepath F
.18667 .06854 0 2.5 Mabsadd m
.18667 .06854 -0.56518 .7779 Mabsadd L
.18667 .06854 -2.37764 .77254 Mabsadd L
.18667 .06854 -0.91448 -0.29713 Mabsadd L
.18667 .06854 -1.46946 -2.02254 Mabsadd L
.18667 .06854 0 -0.96154 Mabsadd L
.18667 .06854 1.46946 -2.02254 Mabsadd L
.18667 .06854 .91448 -0.29713 Mabsadd L
.18667 .06854 2.37764 .77254 Mabsadd L
.18667 .06854 .56518 .7779 Mabsadd L
.18667 .06854 0 2.5 Mabsadd L closepath F
.18762 .06365 0 2.5 Mabsadd m
.18762 .06365 -0.56518 .7779 Mabsadd L
.18762 .06365 -2.37764 .77254 Mabsadd L
.18762 .06365 -0.91448 -0.29713 Mabsadd L
.18762 .06365 -1.46946 -2.02254 Mabsadd L
.18762 .06365 0 -0.96154 Mabsadd L
.18762 .06365 1.46946 -2.02254 Mabsadd L
.18762 .06365 .91448 -0.29713 Mabsadd L
.18762 .06365 2.37764 .77254 Mabsadd L
.18762 .06365 .56518 .7779 Mabsadd L
.18762 .06365 0 2.5 Mabsadd L closepath F
.18857 .06845 0 2.5 Mabsadd m
.18857 .06845 -0.56518 .7779 Mabsadd L
.18857 .06845 -2.37764 .77254 Mabsadd L
.18857 .06845 -0.91448 -0.29713 Mabsadd L
.18857 .06845 -1.46946 -2.02254 Mabsadd L
.18857 .06845 0 -0.96154 Mabsadd L
.18857 .06845 1.46946 -2.02254 Mabsadd L
.18857 .06845 .91448 -0.29713 Mabsadd L
.18857 .06845 2.37764 .77254 Mabsadd L
.18857 .06845 .56518 .7779 Mabsadd L
.18857 .06845 0 2.5 Mabsadd L closepath F
.18952 .0837 0 2.5 Mabsadd m
.18952 .0837 -0.56518 .7779 Mabsadd L
.18952 .0837 -2.37764 .77254 Mabsadd L
.18952 .0837 -0.91448 -0.29713 Mabsadd L
.18952 .0837 -1.46946 -2.02254 Mabsadd L
.18952 .0837 0 -0.96154 Mabsadd L
.18952 .0837 1.46946 -2.02254 Mabsadd L
.18952 .0837 .91448 -0.29713 Mabsadd L
.18952 .0837 2.37764 .77254 Mabsadd L
.18952 .0837 .56518 .7779 Mabsadd L
.18952 .0837 0 2.5 Mabsadd L closepath F
.19048 .08374 0 2.5 Mabsadd m
.19048 .08374 -0.56518 .7779 Mabsadd L
.19048 .08374 -2.37764 .77254 Mabsadd L
.19048 .08374 -0.91448 -0.29713 Mabsadd L
.19048 .08374 -1.46946 -2.02254 Mabsadd L
.19048 .08374 0 -0.96154 Mabsadd L
.19048 .08374 1.46946 -2.02254 Mabsadd L
.19048 .08374 .91448 -0.29713 Mabsadd L
.19048 .08374 2.37764 .77254 Mabsadd L
.19048 .08374 .56518 .7779 Mabsadd L
.19048 .08374 0 2.5 Mabsadd L closepath F
.19143 .08384 0 2.5 Mabsadd m
.19143 .08384 -0.56518 .7779 Mabsadd L
.19143 .08384 -2.37764 .77254 Mabsadd L
.19143 .08384 -0.91448 -0.29713 Mabsadd L
.19143 .08384 -1.46946 -2.02254 Mabsadd L
.19143 .08384 0 -0.96154 Mabsadd L
.19143 .08384 1.46946 -2.02254 Mabsadd L
.19143 .08384 .91448 -0.29713 Mabsadd L
.19143 .08384 2.37764 .77254 Mabsadd L
.19143 .08384 .56518 .7779 Mabsadd L
.19143 .08384 0 2.5 Mabsadd L closepath F
.19238 .08387 0 2.5 Mabsadd m
.19238 .08387 -0.56518 .7779 Mabsadd L
.19238 .08387 -2.37764 .77254 Mabsadd L
.19238 .08387 -0.91448 -0.29713 Mabsadd L
.19238 .08387 -1.46946 -2.02254 Mabsadd L
.19238 .08387 0 -0.96154 Mabsadd L
.19238 .08387 1.46946 -2.02254 Mabsadd L
.19238 .08387 .91448 -0.29713 Mabsadd L
.19238 .08387 2.37764 .77254 Mabsadd L
.19238 .08387 .56518 .7779 Mabsadd L
.19238 .08387 0 2.5 Mabsadd L closepath F
.19333 .08382 0 2.5 Mabsadd m
.19333 .08382 -0.56518 .7779 Mabsadd L
.19333 .08382 -2.37764 .77254 Mabsadd L
.19333 .08382 -0.91448 -0.29713 Mabsadd L
.19333 .08382 -1.46946 -2.02254 Mabsadd L
.19333 .08382 0 -0.96154 Mabsadd L
.19333 .08382 1.46946 -2.02254 Mabsadd L
.19333 .08382 .91448 -0.29713 Mabsadd L
.19333 .08382 2.37764 .77254 Mabsadd L
.19333 .08382 .56518 .7779 Mabsadd L
.19333 .08382 0 2.5 Mabsadd L closepath F
.19429 .08372 0 2.5 Mabsadd m
.19429 .08372 -0.56518 .7779 Mabsadd L
.19429 .08372 -2.37764 .77254 Mabsadd L
.19429 .08372 -0.91448 -0.29713 Mabsadd L
.19429 .08372 -1.46946 -2.02254 Mabsadd L
.19429 .08372 0 -0.96154 Mabsadd L
.19429 .08372 1.46946 -2.02254 Mabsadd L
.19429 .08372 .91448 -0.29713 Mabsadd L
.19429 .08372 2.37764 .77254 Mabsadd L
.19429 .08372 .56518 .7779 Mabsadd L
.19429 .08372 0 2.5 Mabsadd L closepath F
.19524 .08422 0 2.5 Mabsadd m
.19524 .08422 -0.56518 .7779 Mabsadd L
.19524 .08422 -2.37764 .77254 Mabsadd L
.19524 .08422 -0.91448 -0.29713 Mabsadd L
.19524 .08422 -1.46946 -2.02254 Mabsadd L
.19524 .08422 0 -0.96154 Mabsadd L
.19524 .08422 1.46946 -2.02254 Mabsadd L
.19524 .08422 .91448 -0.29713 Mabsadd L
.19524 .08422 2.37764 .77254 Mabsadd L
.19524 .08422 .56518 .7779 Mabsadd L
.19524 .08422 0 2.5 Mabsadd L closepath F
.19619 .08577 0 2.5 Mabsadd m
.19619 .08577 -0.56518 .7779 Mabsadd L
.19619 .08577 -2.37764 .77254 Mabsadd L
.19619 .08577 -0.91448 -0.29713 Mabsadd L
.19619 .08577 -1.46946 -2.02254 Mabsadd L
.19619 .08577 0 -0.96154 Mabsadd L
.19619 .08577 1.46946 -2.02254 Mabsadd L
.19619 .08577 .91448 -0.29713 Mabsadd L
.19619 .08577 2.37764 .77254 Mabsadd L
.19619 .08577 .56518 .7779 Mabsadd L
.19619 .08577 0 2.5 Mabsadd L closepath F
.19714 .08586 0 2.5 Mabsadd m
.19714 .08586 -0.56518 .7779 Mabsadd L
.19714 .08586 -2.37764 .77254 Mabsadd L
.19714 .08586 -0.91448 -0.29713 Mabsadd L
.19714 .08586 -1.46946 -2.02254 Mabsadd L
.19714 .08586 0 -0.96154 Mabsadd L
.19714 .08586 1.46946 -2.02254 Mabsadd L
.19714 .08586 .91448 -0.29713 Mabsadd L
.19714 .08586 2.37764 .77254 Mabsadd L
.19714 .08586 .56518 .7779 Mabsadd L
.19714 .08586 0 2.5 Mabsadd L closepath F
.1981 .08616 0 2.5 Mabsadd m
.1981 .08616 -0.56518 .7779 Mabsadd L
.1981 .08616 -2.37764 .77254 Mabsadd L
.1981 .08616 -0.91448 -0.29713 Mabsadd L
.1981 .08616 -1.46946 -2.02254 Mabsadd L
.1981 .08616 0 -0.96154 Mabsadd L
.1981 .08616 1.46946 -2.02254 Mabsadd L
.1981 .08616 .91448 -0.29713 Mabsadd L
.1981 .08616 2.37764 .77254 Mabsadd L
.1981 .08616 .56518 .7779 Mabsadd L
.1981 .08616 0 2.5 Mabsadd L closepath F
.19905 .08618 0 2.5 Mabsadd m
.19905 .08618 -0.56518 .7779 Mabsadd L
.19905 .08618 -2.37764 .77254 Mabsadd L
.19905 .08618 -0.91448 -0.29713 Mabsadd L
.19905 .08618 -1.46946 -2.02254 Mabsadd L
.19905 .08618 0 -0.96154 Mabsadd L
.19905 .08618 1.46946 -2.02254 Mabsadd L
.19905 .08618 .91448 -0.29713 Mabsadd L
.19905 .08618 2.37764 .77254 Mabsadd L
.19905 .08618 .56518 .7779 Mabsadd L
.19905 .08618 0 2.5 Mabsadd L closepath F
.2 .08625 0 2.5 Mabsadd m
.2 .08625 -0.56518 .7779 Mabsadd L
.2 .08625 -2.37764 .77254 Mabsadd L
.2 .08625 -0.91448 -0.29713 Mabsadd L
.2 .08625 -1.46946 -2.02254 Mabsadd L
.2 .08625 0 -0.96154 Mabsadd L
.2 .08625 1.46946 -2.02254 Mabsadd L
.2 .08625 .91448 -0.29713 Mabsadd L
.2 .08625 2.37764 .77254 Mabsadd L
.2 .08625 .56518 .7779 Mabsadd L
.2 .08625 0 2.5 Mabsadd L closepath F
.20095 .08637 0 2.5 Mabsadd m
.20095 .08637 -0.56518 .7779 Mabsadd L
.20095 .08637 -2.37764 .77254 Mabsadd L
.20095 .08637 -0.91448 -0.29713 Mabsadd L
.20095 .08637 -1.46946 -2.02254 Mabsadd L
.20095 .08637 0 -0.96154 Mabsadd L
.20095 .08637 1.46946 -2.02254 Mabsadd L
.20095 .08637 .91448 -0.29713 Mabsadd L
.20095 .08637 2.37764 .77254 Mabsadd L
.20095 .08637 .56518 .7779 Mabsadd L
.20095 .08637 0 2.5 Mabsadd L closepath F
.2019 .08638 0 2.5 Mabsadd m
.2019 .08638 -0.56518 .7779 Mabsadd L
.2019 .08638 -2.37764 .77254 Mabsadd L
.2019 .08638 -0.91448 -0.29713 Mabsadd L
.2019 .08638 -1.46946 -2.02254 Mabsadd L
.2019 .08638 0 -0.96154 Mabsadd L
.2019 .08638 1.46946 -2.02254 Mabsadd L
.2019 .08638 .91448 -0.29713 Mabsadd L
.2019 .08638 2.37764 .77254 Mabsadd L
.2019 .08638 .56518 .7779 Mabsadd L
.2019 .08638 0 2.5 Mabsadd L closepath F
.20286 .08639 0 2.5 Mabsadd m
.20286 .08639 -0.56518 .7779 Mabsadd L
.20286 .08639 -2.37764 .77254 Mabsadd L
.20286 .08639 -0.91448 -0.29713 Mabsadd L
.20286 .08639 -1.46946 -2.02254 Mabsadd L
.20286 .08639 0 -0.96154 Mabsadd L
.20286 .08639 1.46946 -2.02254 Mabsadd L
.20286 .08639 .91448 -0.29713 Mabsadd L
.20286 .08639 2.37764 .77254 Mabsadd L
.20286 .08639 .56518 .7779 Mabsadd L
.20286 .08639 0 2.5 Mabsadd L closepath F
.20381 .08641 0 2.5 Mabsadd m
.20381 .08641 -0.56518 .7779 Mabsadd L
.20381 .08641 -2.37764 .77254 Mabsadd L
.20381 .08641 -0.91448 -0.29713 Mabsadd L
.20381 .08641 -1.46946 -2.02254 Mabsadd L
.20381 .08641 0 -0.96154 Mabsadd L
.20381 .08641 1.46946 -2.02254 Mabsadd L
.20381 .08641 .91448 -0.29713 Mabsadd L
.20381 .08641 2.37764 .77254 Mabsadd L
.20381 .08641 .56518 .7779 Mabsadd L
.20381 .08641 0 2.5 Mabsadd L closepath F
.20476 .08641 0 2.5 Mabsadd m
.20476 .08641 -0.56518 .7779 Mabsadd L
.20476 .08641 -2.37764 .77254 Mabsadd L
.20476 .08641 -0.91448 -0.29713 Mabsadd L
.20476 .08641 -1.46946 -2.02254 Mabsadd L
.20476 .08641 0 -0.96154 Mabsadd L
.20476 .08641 1.46946 -2.02254 Mabsadd L
.20476 .08641 .91448 -0.29713 Mabsadd L
.20476 .08641 2.37764 .77254 Mabsadd L
.20476 .08641 .56518 .7779 Mabsadd L
.20476 .08641 0 2.5 Mabsadd L closepath F
.20571 .08641 0 2.5 Mabsadd m
.20571 .08641 -0.56518 .7779 Mabsadd L
.20571 .08641 -2.37764 .77254 Mabsadd L
.20571 .08641 -0.91448 -0.29713 Mabsadd L
.20571 .08641 -1.46946 -2.02254 Mabsadd L
.20571 .08641 0 -0.96154 Mabsadd L
.20571 .08641 1.46946 -2.02254 Mabsadd L
.20571 .08641 .91448 -0.29713 Mabsadd L
.20571 .08641 2.37764 .77254 Mabsadd L
.20571 .08641 .56518 .7779 Mabsadd L
.20571 .08641 0 2.5 Mabsadd L closepath F
.20667 .08642 0 2.5 Mabsadd m
.20667 .08642 -0.56518 .7779 Mabsadd L
.20667 .08642 -2.37764 .77254 Mabsadd L
.20667 .08642 -0.91448 -0.29713 Mabsadd L
.20667 .08642 -1.46946 -2.02254 Mabsadd L
.20667 .08642 0 -0.96154 Mabsadd L
.20667 .08642 1.46946 -2.02254 Mabsadd L
.20667 .08642 .91448 -0.29713 Mabsadd L
.20667 .08642 2.37764 .77254 Mabsadd L
.20667 .08642 .56518 .7779 Mabsadd L
.20667 .08642 0 2.5 Mabsadd L closepath F
.20762 .08642 0 2.5 Mabsadd m
.20762 .08642 -0.56518 .7779 Mabsadd L
.20762 .08642 -2.37764 .77254 Mabsadd L
.20762 .08642 -0.91448 -0.29713 Mabsadd L
.20762 .08642 -1.46946 -2.02254 Mabsadd L
.20762 .08642 0 -0.96154 Mabsadd L
.20762 .08642 1.46946 -2.02254 Mabsadd L
.20762 .08642 .91448 -0.29713 Mabsadd L
.20762 .08642 2.37764 .77254 Mabsadd L
.20762 .08642 .56518 .7779 Mabsadd L
.20762 .08642 0 2.5 Mabsadd L closepath F
.20857 .08643 0 2.5 Mabsadd m
.20857 .08643 -0.56518 .7779 Mabsadd L
.20857 .08643 -2.37764 .77254 Mabsadd L
.20857 .08643 -0.91448 -0.29713 Mabsadd L
.20857 .08643 -1.46946 -2.02254 Mabsadd L
.20857 .08643 0 -0.96154 Mabsadd L
.20857 .08643 1.46946 -2.02254 Mabsadd L
.20857 .08643 .91448 -0.29713 Mabsadd L
.20857 .08643 2.37764 .77254 Mabsadd L
.20857 .08643 .56518 .7779 Mabsadd L
.20857 .08643 0 2.5 Mabsadd L closepath F
.20952 .08643 0 2.5 Mabsadd m
.20952 .08643 -0.56518 .7779 Mabsadd L
.20952 .08643 -2.37764 .77254 Mabsadd L
.20952 .08643 -0.91448 -0.29713 Mabsadd L
.20952 .08643 -1.46946 -2.02254 Mabsadd L
.20952 .08643 0 -0.96154 Mabsadd L
.20952 .08643 1.46946 -2.02254 Mabsadd L
.20952 .08643 .91448 -0.29713 Mabsadd L
.20952 .08643 2.37764 .77254 Mabsadd L
.20952 .08643 .56518 .7779 Mabsadd L
.20952 .08643 0 2.5 Mabsadd L closepath F
.21048 .08643 0 2.5 Mabsadd m
.21048 .08643 -0.56518 .7779 Mabsadd L
.21048 .08643 -2.37764 .77254 Mabsadd L
.21048 .08643 -0.91448 -0.29713 Mabsadd L
.21048 .08643 -1.46946 -2.02254 Mabsadd L
.21048 .08643 0 -0.96154 Mabsadd L
.21048 .08643 1.46946 -2.02254 Mabsadd L
.21048 .08643 .91448 -0.29713 Mabsadd L
.21048 .08643 2.37764 .77254 Mabsadd L
.21048 .08643 .56518 .7779 Mabsadd L
.21048 .08643 0 2.5 Mabsadd L closepath F
.21143 .08643 0 2.5 Mabsadd m
.21143 .08643 -0.56518 .7779 Mabsadd L
.21143 .08643 -2.37764 .77254 Mabsadd L
.21143 .08643 -0.91448 -0.29713 Mabsadd L
.21143 .08643 -1.46946 -2.02254 Mabsadd L
.21143 .08643 0 -0.96154 Mabsadd L
.21143 .08643 1.46946 -2.02254 Mabsadd L
.21143 .08643 .91448 -0.29713 Mabsadd L
.21143 .08643 2.37764 .77254 Mabsadd L
.21143 .08643 .56518 .7779 Mabsadd L
.21143 .08643 0 2.5 Mabsadd L closepath F
.21238 .08643 0 2.5 Mabsadd m
.21238 .08643 -0.56518 .7779 Mabsadd L
.21238 .08643 -2.37764 .77254 Mabsadd L
.21238 .08643 -0.91448 -0.29713 Mabsadd L
.21238 .08643 -1.46946 -2.02254 Mabsadd L
.21238 .08643 0 -0.96154 Mabsadd L
.21238 .08643 1.46946 -2.02254 Mabsadd L
.21238 .08643 .91448 -0.29713 Mabsadd L
.21238 .08643 2.37764 .77254 Mabsadd L
.21238 .08643 .56518 .7779 Mabsadd L
.21238 .08643 0 2.5 Mabsadd L closepath F
.21333 .08643 0 2.5 Mabsadd m
.21333 .08643 -0.56518 .7779 Mabsadd L
.21333 .08643 -2.37764 .77254 Mabsadd L
.21333 .08643 -0.91448 -0.29713 Mabsadd L
.21333 .08643 -1.46946 -2.02254 Mabsadd L
.21333 .08643 0 -0.96154 Mabsadd L
.21333 .08643 1.46946 -2.02254 Mabsadd L
.21333 .08643 .91448 -0.29713 Mabsadd L
.21333 .08643 2.37764 .77254 Mabsadd L
.21333 .08643 .56518 .7779 Mabsadd L
.21333 .08643 0 2.5 Mabsadd L closepath F
.21429 .08642 0 2.5 Mabsadd m
.21429 .08642 -0.56518 .7779 Mabsadd L
.21429 .08642 -2.37764 .77254 Mabsadd L
.21429 .08642 -0.91448 -0.29713 Mabsadd L
.21429 .08642 -1.46946 -2.02254 Mabsadd L
.21429 .08642 0 -0.96154 Mabsadd L
.21429 .08642 1.46946 -2.02254 Mabsadd L
.21429 .08642 .91448 -0.29713 Mabsadd L
.21429 .08642 2.37764 .77254 Mabsadd L
.21429 .08642 .56518 .7779 Mabsadd L
.21429 .08642 0 2.5 Mabsadd L closepath F
.21524 .08641 0 2.5 Mabsadd m
.21524 .08641 -0.56518 .7779 Mabsadd L
.21524 .08641 -2.37764 .77254 Mabsadd L
.21524 .08641 -0.91448 -0.29713 Mabsadd L
.21524 .08641 -1.46946 -2.02254 Mabsadd L
.21524 .08641 0 -0.96154 Mabsadd L
.21524 .08641 1.46946 -2.02254 Mabsadd L
.21524 .08641 .91448 -0.29713 Mabsadd L
.21524 .08641 2.37764 .77254 Mabsadd L
.21524 .08641 .56518 .7779 Mabsadd L
.21524 .08641 0 2.5 Mabsadd L closepath F
.21619 .08641 0 2.5 Mabsadd m
.21619 .08641 -0.56518 .7779 Mabsadd L
.21619 .08641 -2.37764 .77254 Mabsadd L
.21619 .08641 -0.91448 -0.29713 Mabsadd L
.21619 .08641 -1.46946 -2.02254 Mabsadd L
.21619 .08641 0 -0.96154 Mabsadd L
.21619 .08641 1.46946 -2.02254 Mabsadd L
.21619 .08641 .91448 -0.29713 Mabsadd L
.21619 .08641 2.37764 .77254 Mabsadd L
.21619 .08641 .56518 .7779 Mabsadd L
.21619 .08641 0 2.5 Mabsadd L closepath F
.21714 .08641 0 2.5 Mabsadd m
.21714 .08641 -0.56518 .7779 Mabsadd L
.21714 .08641 -2.37764 .77254 Mabsadd L
.21714 .08641 -0.91448 -0.29713 Mabsadd L
.21714 .08641 -1.46946 -2.02254 Mabsadd L
.21714 .08641 0 -0.96154 Mabsadd L
.21714 .08641 1.46946 -2.02254 Mabsadd L
.21714 .08641 .91448 -0.29713 Mabsadd L
.21714 .08641 2.37764 .77254 Mabsadd L
.21714 .08641 .56518 .7779 Mabsadd L
.21714 .08641 0 2.5 Mabsadd L closepath F
.2181 .0864 0 2.5 Mabsadd m
.2181 .0864 -0.56518 .7779 Mabsadd L
.2181 .0864 -2.37764 .77254 Mabsadd L
.2181 .0864 -0.91448 -0.29713 Mabsadd L
.2181 .0864 -1.46946 -2.02254 Mabsadd L
.2181 .0864 0 -0.96154 Mabsadd L
.2181 .0864 1.46946 -2.02254 Mabsadd L
.2181 .0864 .91448 -0.29713 Mabsadd L
.2181 .0864 2.37764 .77254 Mabsadd L
.2181 .0864 .56518 .7779 Mabsadd L
.2181 .0864 0 2.5 Mabsadd L closepath F
.21905 .08639 0 2.5 Mabsadd m
.21905 .08639 -0.56518 .7779 Mabsadd L
.21905 .08639 -2.37764 .77254 Mabsadd L
.21905 .08639 -0.91448 -0.29713 Mabsadd L
.21905 .08639 -1.46946 -2.02254 Mabsadd L
.21905 .08639 0 -0.96154 Mabsadd L
.21905 .08639 1.46946 -2.02254 Mabsadd L
.21905 .08639 .91448 -0.29713 Mabsadd L
.21905 .08639 2.37764 .77254 Mabsadd L
.21905 .08639 .56518 .7779 Mabsadd L
.21905 .08639 0 2.5 Mabsadd L closepath F
.22 .08639 0 2.5 Mabsadd m
.22 .08639 -0.56518 .7779 Mabsadd L
.22 .08639 -2.37764 .77254 Mabsadd L
.22 .08639 -0.91448 -0.29713 Mabsadd L
.22 .08639 -1.46946 -2.02254 Mabsadd L
.22 .08639 0 -0.96154 Mabsadd L
.22 .08639 1.46946 -2.02254 Mabsadd L
.22 .08639 .91448 -0.29713 Mabsadd L
.22 .08639 2.37764 .77254 Mabsadd L
.22 .08639 .56518 .7779 Mabsadd L
.22 .08639 0 2.5 Mabsadd L closepath F
.22095 .08638 0 2.5 Mabsadd m
.22095 .08638 -0.56518 .7779 Mabsadd L
.22095 .08638 -2.37764 .77254 Mabsadd L
.22095 .08638 -0.91448 -0.29713 Mabsadd L
.22095 .08638 -1.46946 -2.02254 Mabsadd L
.22095 .08638 0 -0.96154 Mabsadd L
.22095 .08638 1.46946 -2.02254 Mabsadd L
.22095 .08638 .91448 -0.29713 Mabsadd L
.22095 .08638 2.37764 .77254 Mabsadd L
.22095 .08638 .56518 .7779 Mabsadd L
.22095 .08638 0 2.5 Mabsadd L closepath F
.2219 .08634 0 2.5 Mabsadd m
.2219 .08634 -0.56518 .7779 Mabsadd L
.2219 .08634 -2.37764 .77254 Mabsadd L
.2219 .08634 -0.91448 -0.29713 Mabsadd L
.2219 .08634 -1.46946 -2.02254 Mabsadd L
.2219 .08634 0 -0.96154 Mabsadd L
.2219 .08634 1.46946 -2.02254 Mabsadd L
.2219 .08634 .91448 -0.29713 Mabsadd L
.2219 .08634 2.37764 .77254 Mabsadd L
.2219 .08634 .56518 .7779 Mabsadd L
.2219 .08634 0 2.5 Mabsadd L closepath F
.22286 .08634 0 2.5 Mabsadd m
.22286 .08634 -0.56518 .7779 Mabsadd L
.22286 .08634 -2.37764 .77254 Mabsadd L
.22286 .08634 -0.91448 -0.29713 Mabsadd L
.22286 .08634 -1.46946 -2.02254 Mabsadd L
.22286 .08634 0 -0.96154 Mabsadd L
.22286 .08634 1.46946 -2.02254 Mabsadd L
.22286 .08634 .91448 -0.29713 Mabsadd L
.22286 .08634 2.37764 .77254 Mabsadd L
.22286 .08634 .56518 .7779 Mabsadd L
.22286 .08634 0 2.5 Mabsadd L closepath F
.22381 .08629 0 2.5 Mabsadd m
.22381 .08629 -0.56518 .7779 Mabsadd L
.22381 .08629 -2.37764 .77254 Mabsadd L
.22381 .08629 -0.91448 -0.29713 Mabsadd L
.22381 .08629 -1.46946 -2.02254 Mabsadd L
.22381 .08629 0 -0.96154 Mabsadd L
.22381 .08629 1.46946 -2.02254 Mabsadd L
.22381 .08629 .91448 -0.29713 Mabsadd L
.22381 .08629 2.37764 .77254 Mabsadd L
.22381 .08629 .56518 .7779 Mabsadd L
.22381 .08629 0 2.5 Mabsadd L closepath F
.22476 .08609 0 2.5 Mabsadd m
.22476 .08609 -0.56518 .7779 Mabsadd L
.22476 .08609 -2.37764 .77254 Mabsadd L
.22476 .08609 -0.91448 -0.29713 Mabsadd L
.22476 .08609 -1.46946 -2.02254 Mabsadd L
.22476 .08609 0 -0.96154 Mabsadd L
.22476 .08609 1.46946 -2.02254 Mabsadd L
.22476 .08609 .91448 -0.29713 Mabsadd L
.22476 .08609 2.37764 .77254 Mabsadd L
.22476 .08609 .56518 .7779 Mabsadd L
.22476 .08609 0 2.5 Mabsadd L closepath F
.22571 .08605 0 2.5 Mabsadd m
.22571 .08605 -0.56518 .7779 Mabsadd L
.22571 .08605 -2.37764 .77254 Mabsadd L
.22571 .08605 -0.91448 -0.29713 Mabsadd L
.22571 .08605 -1.46946 -2.02254 Mabsadd L
.22571 .08605 0 -0.96154 Mabsadd L
.22571 .08605 1.46946 -2.02254 Mabsadd L
.22571 .08605 .91448 -0.29713 Mabsadd L
.22571 .08605 2.37764 .77254 Mabsadd L
.22571 .08605 .56518 .7779 Mabsadd L
.22571 .08605 0 2.5 Mabsadd L closepath F
.22667 .08564 0 2.5 Mabsadd m
.22667 .08564 -0.56518 .7779 Mabsadd L
.22667 .08564 -2.37764 .77254 Mabsadd L
.22667 .08564 -0.91448 -0.29713 Mabsadd L
.22667 .08564 -1.46946 -2.02254 Mabsadd L
.22667 .08564 0 -0.96154 Mabsadd L
.22667 .08564 1.46946 -2.02254 Mabsadd L
.22667 .08564 .91448 -0.29713 Mabsadd L
.22667 .08564 2.37764 .77254 Mabsadd L
.22667 .08564 .56518 .7779 Mabsadd L
.22667 .08564 0 2.5 Mabsadd L closepath F
.22762 .08543 0 2.5 Mabsadd m
.22762 .08543 -0.56518 .7779 Mabsadd L
.22762 .08543 -2.37764 .77254 Mabsadd L
.22762 .08543 -0.91448 -0.29713 Mabsadd L
.22762 .08543 -1.46946 -2.02254 Mabsadd L
.22762 .08543 0 -0.96154 Mabsadd L
.22762 .08543 1.46946 -2.02254 Mabsadd L
.22762 .08543 .91448 -0.29713 Mabsadd L
.22762 .08543 2.37764 .77254 Mabsadd L
.22762 .08543 .56518 .7779 Mabsadd L
.22762 .08543 0 2.5 Mabsadd L closepath F
.22857 .08315 0 2.5 Mabsadd m
.22857 .08315 -0.56518 .7779 Mabsadd L
.22857 .08315 -2.37764 .77254 Mabsadd L
.22857 .08315 -0.91448 -0.29713 Mabsadd L
.22857 .08315 -1.46946 -2.02254 Mabsadd L
.22857 .08315 0 -0.96154 Mabsadd L
.22857 .08315 1.46946 -2.02254 Mabsadd L
.22857 .08315 .91448 -0.29713 Mabsadd L
.22857 .08315 2.37764 .77254 Mabsadd L
.22857 .08315 .56518 .7779 Mabsadd L
.22857 .08315 0 2.5 Mabsadd L closepath F
.22952 .08166 0 2.5 Mabsadd m
.22952 .08166 -0.56518 .7779 Mabsadd L
.22952 .08166 -2.37764 .77254 Mabsadd L
.22952 .08166 -0.91448 -0.29713 Mabsadd L
.22952 .08166 -1.46946 -2.02254 Mabsadd L
.22952 .08166 0 -0.96154 Mabsadd L
.22952 .08166 1.46946 -2.02254 Mabsadd L
.22952 .08166 .91448 -0.29713 Mabsadd L
.22952 .08166 2.37764 .77254 Mabsadd L
.22952 .08166 .56518 .7779 Mabsadd L
.22952 .08166 0 2.5 Mabsadd L closepath F
.23048 .06418 0 2.5 Mabsadd m
.23048 .06418 -0.56518 .7779 Mabsadd L
.23048 .06418 -2.37764 .77254 Mabsadd L
.23048 .06418 -0.91448 -0.29713 Mabsadd L
.23048 .06418 -1.46946 -2.02254 Mabsadd L
.23048 .06418 0 -0.96154 Mabsadd L
.23048 .06418 1.46946 -2.02254 Mabsadd L
.23048 .06418 .91448 -0.29713 Mabsadd L
.23048 .06418 2.37764 .77254 Mabsadd L
.23048 .06418 .56518 .7779 Mabsadd L
.23048 .06418 0 2.5 Mabsadd L closepath F
.23143 .04931 0 2.5 Mabsadd m
.23143 .04931 -0.56518 .7779 Mabsadd L
.23143 .04931 -2.37764 .77254 Mabsadd L
.23143 .04931 -0.91448 -0.29713 Mabsadd L
.23143 .04931 -1.46946 -2.02254 Mabsadd L
.23143 .04931 0 -0.96154 Mabsadd L
.23143 .04931 1.46946 -2.02254 Mabsadd L
.23143 .04931 .91448 -0.29713 Mabsadd L
.23143 .04931 2.37764 .77254 Mabsadd L
.23143 .04931 .56518 .7779 Mabsadd L
.23143 .04931 0 2.5 Mabsadd L closepath F
.23238 .05345 0 2.5 Mabsadd m
.23238 .05345 -0.56518 .7779 Mabsadd L
.23238 .05345 -2.37764 .77254 Mabsadd L
.23238 .05345 -0.91448 -0.29713 Mabsadd L
.23238 .05345 -1.46946 -2.02254 Mabsadd L
.23238 .05345 0 -0.96154 Mabsadd L
.23238 .05345 1.46946 -2.02254 Mabsadd L
.23238 .05345 .91448 -0.29713 Mabsadd L
.23238 .05345 2.37764 .77254 Mabsadd L
.23238 .05345 .56518 .7779 Mabsadd L
.23238 .05345 0 2.5 Mabsadd L closepath F
.23333 .05341 0 2.5 Mabsadd m
.23333 .05341 -0.56518 .7779 Mabsadd L
.23333 .05341 -2.37764 .77254 Mabsadd L
.23333 .05341 -0.91448 -0.29713 Mabsadd L
.23333 .05341 -1.46946 -2.02254 Mabsadd L
.23333 .05341 0 -0.96154 Mabsadd L
.23333 .05341 1.46946 -2.02254 Mabsadd L
.23333 .05341 .91448 -0.29713 Mabsadd L
.23333 .05341 2.37764 .77254 Mabsadd L
.23333 .05341 .56518 .7779 Mabsadd L
.23333 .05341 0 2.5 Mabsadd L closepath F
.23429 .05028 0 2.5 Mabsadd m
.23429 .05028 -0.56518 .7779 Mabsadd L
.23429 .05028 -2.37764 .77254 Mabsadd L
.23429 .05028 -0.91448 -0.29713 Mabsadd L
.23429 .05028 -1.46946 -2.02254 Mabsadd L
.23429 .05028 0 -0.96154 Mabsadd L
.23429 .05028 1.46946 -2.02254 Mabsadd L
.23429 .05028 .91448 -0.29713 Mabsadd L
.23429 .05028 2.37764 .77254 Mabsadd L
.23429 .05028 .56518 .7779 Mabsadd L
.23429 .05028 0 2.5 Mabsadd L closepath F
.23524 .04415 0 2.5 Mabsadd m
.23524 .04415 -0.56518 .7779 Mabsadd L
.23524 .04415 -2.37764 .77254 Mabsadd L
.23524 .04415 -0.91448 -0.29713 Mabsadd L
.23524 .04415 -1.46946 -2.02254 Mabsadd L
.23524 .04415 0 -0.96154 Mabsadd L
.23524 .04415 1.46946 -2.02254 Mabsadd L
.23524 .04415 .91448 -0.29713 Mabsadd L
.23524 .04415 2.37764 .77254 Mabsadd L
.23524 .04415 .56518 .7779 Mabsadd L
.23524 .04415 0 2.5 Mabsadd L closepath F
.23619 .04386 0 2.5 Mabsadd m
.23619 .04386 -0.56518 .7779 Mabsadd L
.23619 .04386 -2.37764 .77254 Mabsadd L
.23619 .04386 -0.91448 -0.29713 Mabsadd L
.23619 .04386 -1.46946 -2.02254 Mabsadd L
.23619 .04386 0 -0.96154 Mabsadd L
.23619 .04386 1.46946 -2.02254 Mabsadd L
.23619 .04386 .91448 -0.29713 Mabsadd L
.23619 .04386 2.37764 .77254 Mabsadd L
.23619 .04386 .56518 .7779 Mabsadd L
.23619 .04386 0 2.5 Mabsadd L closepath F
.23714 .04324 0 2.5 Mabsadd m
.23714 .04324 -0.56518 .7779 Mabsadd L
.23714 .04324 -2.37764 .77254 Mabsadd L
.23714 .04324 -0.91448 -0.29713 Mabsadd L
.23714 .04324 -1.46946 -2.02254 Mabsadd L
.23714 .04324 0 -0.96154 Mabsadd L
.23714 .04324 1.46946 -2.02254 Mabsadd L
.23714 .04324 .91448 -0.29713 Mabsadd L
.23714 .04324 2.37764 .77254 Mabsadd L
.23714 .04324 .56518 .7779 Mabsadd L
.23714 .04324 0 2.5 Mabsadd L closepath F
.2381 .03969 0 2.5 Mabsadd m
.2381 .03969 -0.56518 .7779 Mabsadd L
.2381 .03969 -2.37764 .77254 Mabsadd L
.2381 .03969 -0.91448 -0.29713 Mabsadd L
.2381 .03969 -1.46946 -2.02254 Mabsadd L
.2381 .03969 0 -0.96154 Mabsadd L
.2381 .03969 1.46946 -2.02254 Mabsadd L
.2381 .03969 .91448 -0.29713 Mabsadd L
.2381 .03969 2.37764 .77254 Mabsadd L
.2381 .03969 .56518 .7779 Mabsadd L
.2381 .03969 0 2.5 Mabsadd L closepath F
.23905 .03761 0 2.5 Mabsadd m
.23905 .03761 -0.56518 .7779 Mabsadd L
.23905 .03761 -2.37764 .77254 Mabsadd L
.23905 .03761 -0.91448 -0.29713 Mabsadd L
.23905 .03761 -1.46946 -2.02254 Mabsadd L
.23905 .03761 0 -0.96154 Mabsadd L
.23905 .03761 1.46946 -2.02254 Mabsadd L
.23905 .03761 .91448 -0.29713 Mabsadd L
.23905 .03761 2.37764 .77254 Mabsadd L
.23905 .03761 .56518 .7779 Mabsadd L
.23905 .03761 0 2.5 Mabsadd L closepath F
.24 .03787 0 2.5 Mabsadd m
.24 .03787 -0.56518 .7779 Mabsadd L
.24 .03787 -2.37764 .77254 Mabsadd L
.24 .03787 -0.91448 -0.29713 Mabsadd L
.24 .03787 -1.46946 -2.02254 Mabsadd L
.24 .03787 0 -0.96154 Mabsadd L
.24 .03787 1.46946 -2.02254 Mabsadd L
.24 .03787 .91448 -0.29713 Mabsadd L
.24 .03787 2.37764 .77254 Mabsadd L
.24 .03787 .56518 .7779 Mabsadd L
.24 .03787 0 2.5 Mabsadd L closepath F
.24095 .04477 0 2.5 Mabsadd m
.24095 .04477 -0.56518 .7779 Mabsadd L
.24095 .04477 -2.37764 .77254 Mabsadd L
.24095 .04477 -0.91448 -0.29713 Mabsadd L
.24095 .04477 -1.46946 -2.02254 Mabsadd L
.24095 .04477 0 -0.96154 Mabsadd L
.24095 .04477 1.46946 -2.02254 Mabsadd L
.24095 .04477 .91448 -0.29713 Mabsadd L
.24095 .04477 2.37764 .77254 Mabsadd L
.24095 .04477 .56518 .7779 Mabsadd L
.24095 .04477 0 2.5 Mabsadd L closepath F
.2419 .03821 0 2.5 Mabsadd m
.2419 .03821 -0.56518 .7779 Mabsadd L
.2419 .03821 -2.37764 .77254 Mabsadd L
.2419 .03821 -0.91448 -0.29713 Mabsadd L
.2419 .03821 -1.46946 -2.02254 Mabsadd L
.2419 .03821 0 -0.96154 Mabsadd L
.2419 .03821 1.46946 -2.02254 Mabsadd L
.2419 .03821 .91448 -0.29713 Mabsadd L
.2419 .03821 2.37764 .77254 Mabsadd L
.2419 .03821 .56518 .7779 Mabsadd L
.2419 .03821 0 2.5 Mabsadd L closepath F
.24286 .03094 0 2.5 Mabsadd m
.24286 .03094 -0.56518 .7779 Mabsadd L
.24286 .03094 -2.37764 .77254 Mabsadd L
.24286 .03094 -0.91448 -0.29713 Mabsadd L
.24286 .03094 -1.46946 -2.02254 Mabsadd L
.24286 .03094 0 -0.96154 Mabsadd L
.24286 .03094 1.46946 -2.02254 Mabsadd L
.24286 .03094 .91448 -0.29713 Mabsadd L
.24286 .03094 2.37764 .77254 Mabsadd L
.24286 .03094 .56518 .7779 Mabsadd L
.24286 .03094 0 2.5 Mabsadd L closepath F
.24381 .03099 0 2.5 Mabsadd m
.24381 .03099 -0.56518 .7779 Mabsadd L
.24381 .03099 -2.37764 .77254 Mabsadd L
.24381 .03099 -0.91448 -0.29713 Mabsadd L
.24381 .03099 -1.46946 -2.02254 Mabsadd L
.24381 .03099 0 -0.96154 Mabsadd L
.24381 .03099 1.46946 -2.02254 Mabsadd L
.24381 .03099 .91448 -0.29713 Mabsadd L
.24381 .03099 2.37764 .77254 Mabsadd L
.24381 .03099 .56518 .7779 Mabsadd L
.24381 .03099 0 2.5 Mabsadd L closepath F
.24476 .03112 0 2.5 Mabsadd m
.24476 .03112 -0.56518 .7779 Mabsadd L
.24476 .03112 -2.37764 .77254 Mabsadd L
.24476 .03112 -0.91448 -0.29713 Mabsadd L
.24476 .03112 -1.46946 -2.02254 Mabsadd L
.24476 .03112 0 -0.96154 Mabsadd L
.24476 .03112 1.46946 -2.02254 Mabsadd L
.24476 .03112 .91448 -0.29713 Mabsadd L
.24476 .03112 2.37764 .77254 Mabsadd L
.24476 .03112 .56518 .7779 Mabsadd L
.24476 .03112 0 2.5 Mabsadd L closepath F
.24571 .03118 0 2.5 Mabsadd m
.24571 .03118 -0.56518 .7779 Mabsadd L
.24571 .03118 -2.37764 .77254 Mabsadd L
.24571 .03118 -0.91448 -0.29713 Mabsadd L
.24571 .03118 -1.46946 -2.02254 Mabsadd L
.24571 .03118 0 -0.96154 Mabsadd L
.24571 .03118 1.46946 -2.02254 Mabsadd L
.24571 .03118 .91448 -0.29713 Mabsadd L
.24571 .03118 2.37764 .77254 Mabsadd L
.24571 .03118 .56518 .7779 Mabsadd L
.24571 .03118 0 2.5 Mabsadd L closepath F
.24667 .03112 0 2.5 Mabsadd m
.24667 .03112 -0.56518 .7779 Mabsadd L
.24667 .03112 -2.37764 .77254 Mabsadd L
.24667 .03112 -0.91448 -0.29713 Mabsadd L
.24667 .03112 -1.46946 -2.02254 Mabsadd L
.24667 .03112 0 -0.96154 Mabsadd L
.24667 .03112 1.46946 -2.02254 Mabsadd L
.24667 .03112 .91448 -0.29713 Mabsadd L
.24667 .03112 2.37764 .77254 Mabsadd L
.24667 .03112 .56518 .7779 Mabsadd L
.24667 .03112 0 2.5 Mabsadd L closepath F
.24762 .03099 0 2.5 Mabsadd m
.24762 .03099 -0.56518 .7779 Mabsadd L
.24762 .03099 -2.37764 .77254 Mabsadd L
.24762 .03099 -0.91448 -0.29713 Mabsadd L
.24762 .03099 -1.46946 -2.02254 Mabsadd L
.24762 .03099 0 -0.96154 Mabsadd L
.24762 .03099 1.46946 -2.02254 Mabsadd L
.24762 .03099 .91448 -0.29713 Mabsadd L
.24762 .03099 2.37764 .77254 Mabsadd L
.24762 .03099 .56518 .7779 Mabsadd L
.24762 .03099 0 2.5 Mabsadd L closepath F
.24857 .03156 0 2.5 Mabsadd m
.24857 .03156 -0.56518 .7779 Mabsadd L
.24857 .03156 -2.37764 .77254 Mabsadd L
.24857 .03156 -0.91448 -0.29713 Mabsadd L
.24857 .03156 -1.46946 -2.02254 Mabsadd L
.24857 .03156 0 -0.96154 Mabsadd L
.24857 .03156 1.46946 -2.02254 Mabsadd L
.24857 .03156 .91448 -0.29713 Mabsadd L
.24857 .03156 2.37764 .77254 Mabsadd L
.24857 .03156 .56518 .7779 Mabsadd L
.24857 .03156 0 2.5 Mabsadd L closepath F
.24952 .0333 0 2.5 Mabsadd m
.24952 .0333 -0.56518 .7779 Mabsadd L
.24952 .0333 -2.37764 .77254 Mabsadd L
.24952 .0333 -0.91448 -0.29713 Mabsadd L
.24952 .0333 -1.46946 -2.02254 Mabsadd L
.24952 .0333 0 -0.96154 Mabsadd L
.24952 .0333 1.46946 -2.02254 Mabsadd L
.24952 .0333 .91448 -0.29713 Mabsadd L
.24952 .0333 2.37764 .77254 Mabsadd L
.24952 .0333 .56518 .7779 Mabsadd L
.24952 .0333 0 2.5 Mabsadd L closepath F
.25048 .0334 0 2.5 Mabsadd m
.25048 .0334 -0.56518 .7779 Mabsadd L
.25048 .0334 -2.37764 .77254 Mabsadd L
.25048 .0334 -0.91448 -0.29713 Mabsadd L
.25048 .0334 -1.46946 -2.02254 Mabsadd L
.25048 .0334 0 -0.96154 Mabsadd L
.25048 .0334 1.46946 -2.02254 Mabsadd L
.25048 .0334 .91448 -0.29713 Mabsadd L
.25048 .0334 2.37764 .77254 Mabsadd L
.25048 .0334 .56518 .7779 Mabsadd L
.25048 .0334 0 2.5 Mabsadd L closepath F
.25143 .03373 0 2.5 Mabsadd m
.25143 .03373 -0.56518 .7779 Mabsadd L
.25143 .03373 -2.37764 .77254 Mabsadd L
.25143 .03373 -0.91448 -0.29713 Mabsadd L
.25143 .03373 -1.46946 -2.02254 Mabsadd L
.25143 .03373 0 -0.96154 Mabsadd L
.25143 .03373 1.46946 -2.02254 Mabsadd L
.25143 .03373 .91448 -0.29713 Mabsadd L
.25143 .03373 2.37764 .77254 Mabsadd L
.25143 .03373 .56518 .7779 Mabsadd L
.25143 .03373 0 2.5 Mabsadd L closepath F
.25238 .03375 0 2.5 Mabsadd m
.25238 .03375 -0.56518 .7779 Mabsadd L
.25238 .03375 -2.37764 .77254 Mabsadd L
.25238 .03375 -0.91448 -0.29713 Mabsadd L
.25238 .03375 -1.46946 -2.02254 Mabsadd L
.25238 .03375 0 -0.96154 Mabsadd L
.25238 .03375 1.46946 -2.02254 Mabsadd L
.25238 .03375 .91448 -0.29713 Mabsadd L
.25238 .03375 2.37764 .77254 Mabsadd L
.25238 .03375 .56518 .7779 Mabsadd L
.25238 .03375 0 2.5 Mabsadd L closepath F
.25333 .03383 0 2.5 Mabsadd m
.25333 .03383 -0.56518 .7779 Mabsadd L
.25333 .03383 -2.37764 .77254 Mabsadd L
.25333 .03383 -0.91448 -0.29713 Mabsadd L
.25333 .03383 -1.46946 -2.02254 Mabsadd L
.25333 .03383 0 -0.96154 Mabsadd L
.25333 .03383 1.46946 -2.02254 Mabsadd L
.25333 .03383 .91448 -0.29713 Mabsadd L
.25333 .03383 2.37764 .77254 Mabsadd L
.25333 .03383 .56518 .7779 Mabsadd L
.25333 .03383 0 2.5 Mabsadd L closepath F
.25429 .03396 0 2.5 Mabsadd m
.25429 .03396 -0.56518 .7779 Mabsadd L
.25429 .03396 -2.37764 .77254 Mabsadd L
.25429 .03396 -0.91448 -0.29713 Mabsadd L
.25429 .03396 -1.46946 -2.02254 Mabsadd L
.25429 .03396 0 -0.96154 Mabsadd L
.25429 .03396 1.46946 -2.02254 Mabsadd L
.25429 .03396 .91448 -0.29713 Mabsadd L
.25429 .03396 2.37764 .77254 Mabsadd L
.25429 .03396 .56518 .7779 Mabsadd L
.25429 .03396 0 2.5 Mabsadd L closepath F
.25524 .03396 0 2.5 Mabsadd m
.25524 .03396 -0.56518 .7779 Mabsadd L
.25524 .03396 -2.37764 .77254 Mabsadd L
.25524 .03396 -0.91448 -0.29713 Mabsadd L
.25524 .03396 -1.46946 -2.02254 Mabsadd L
.25524 .03396 0 -0.96154 Mabsadd L
.25524 .03396 1.46946 -2.02254 Mabsadd L
.25524 .03396 .91448 -0.29713 Mabsadd L
.25524 .03396 2.37764 .77254 Mabsadd L
.25524 .03396 .56518 .7779 Mabsadd L
.25524 .03396 0 2.5 Mabsadd L closepath F
.25619 .03397 0 2.5 Mabsadd m
.25619 .03397 -0.56518 .7779 Mabsadd L
.25619 .03397 -2.37764 .77254 Mabsadd L
.25619 .03397 -0.91448 -0.29713 Mabsadd L
.25619 .03397 -1.46946 -2.02254 Mabsadd L
.25619 .03397 0 -0.96154 Mabsadd L
.25619 .03397 1.46946 -2.02254 Mabsadd L
.25619 .03397 .91448 -0.29713 Mabsadd L
.25619 .03397 2.37764 .77254 Mabsadd L
.25619 .03397 .56518 .7779 Mabsadd L
.25619 .03397 0 2.5 Mabsadd L closepath F
.25714 .034 0 2.5 Mabsadd m
.25714 .034 -0.56518 .7779 Mabsadd L
.25714 .034 -2.37764 .77254 Mabsadd L
.25714 .034 -0.91448 -0.29713 Mabsadd L
.25714 .034 -1.46946 -2.02254 Mabsadd L
.25714 .034 0 -0.96154 Mabsadd L
.25714 .034 1.46946 -2.02254 Mabsadd L
.25714 .034 .91448 -0.29713 Mabsadd L
.25714 .034 2.37764 .77254 Mabsadd L
.25714 .034 .56518 .7779 Mabsadd L
.25714 .034 0 2.5 Mabsadd L closepath F
.2581 .034 0 2.5 Mabsadd m
.2581 .034 -0.56518 .7779 Mabsadd L
.2581 .034 -2.37764 .77254 Mabsadd L
.2581 .034 -0.91448 -0.29713 Mabsadd L
.2581 .034 -1.46946 -2.02254 Mabsadd L
.2581 .034 0 -0.96154 Mabsadd L
.2581 .034 1.46946 -2.02254 Mabsadd L
.2581 .034 .91448 -0.29713 Mabsadd L
.2581 .034 2.37764 .77254 Mabsadd L
.2581 .034 .56518 .7779 Mabsadd L
.2581 .034 0 2.5 Mabsadd L closepath F
.25905 .034 0 2.5 Mabsadd m
.25905 .034 -0.56518 .7779 Mabsadd L
.25905 .034 -2.37764 .77254 Mabsadd L
.25905 .034 -0.91448 -0.29713 Mabsadd L
.25905 .034 -1.46946 -2.02254 Mabsadd L
.25905 .034 0 -0.96154 Mabsadd L
.25905 .034 1.46946 -2.02254 Mabsadd L
.25905 .034 .91448 -0.29713 Mabsadd L
.25905 .034 2.37764 .77254 Mabsadd L
.25905 .034 .56518 .7779 Mabsadd L
.25905 .034 0 2.5 Mabsadd L closepath F
.26 .03401 0 2.5 Mabsadd m
.26 .03401 -0.56518 .7779 Mabsadd L
.26 .03401 -2.37764 .77254 Mabsadd L
.26 .03401 -0.91448 -0.29713 Mabsadd L
.26 .03401 -1.46946 -2.02254 Mabsadd L
.26 .03401 0 -0.96154 Mabsadd L
.26 .03401 1.46946 -2.02254 Mabsadd L
.26 .03401 .91448 -0.29713 Mabsadd L
.26 .03401 2.37764 .77254 Mabsadd L
.26 .03401 .56518 .7779 Mabsadd L
.26 .03401 0 2.5 Mabsadd L closepath F
.26095 .03401 0 2.5 Mabsadd m
.26095 .03401 -0.56518 .7779 Mabsadd L
.26095 .03401 -2.37764 .77254 Mabsadd L
.26095 .03401 -0.91448 -0.29713 Mabsadd L
.26095 .03401 -1.46946 -2.02254 Mabsadd L
.26095 .03401 0 -0.96154 Mabsadd L
.26095 .03401 1.46946 -2.02254 Mabsadd L
.26095 .03401 .91448 -0.29713 Mabsadd L
.26095 .03401 2.37764 .77254 Mabsadd L
.26095 .03401 .56518 .7779 Mabsadd L
.26095 .03401 0 2.5 Mabsadd L closepath F
.2619 .03401 0 2.5 Mabsadd m
.2619 .03401 -0.56518 .7779 Mabsadd L
.2619 .03401 -2.37764 .77254 Mabsadd L
.2619 .03401 -0.91448 -0.29713 Mabsadd L
.2619 .03401 -1.46946 -2.02254 Mabsadd L
.2619 .03401 0 -0.96154 Mabsadd L
.2619 .03401 1.46946 -2.02254 Mabsadd L
.2619 .03401 .91448 -0.29713 Mabsadd L
.2619 .03401 2.37764 .77254 Mabsadd L
.2619 .03401 .56518 .7779 Mabsadd L
.2619 .03401 0 2.5 Mabsadd L closepath F
.26286 .03401 0 2.5 Mabsadd m
.26286 .03401 -0.56518 .7779 Mabsadd L
.26286 .03401 -2.37764 .77254 Mabsadd L
.26286 .03401 -0.91448 -0.29713 Mabsadd L
.26286 .03401 -1.46946 -2.02254 Mabsadd L
.26286 .03401 0 -0.96154 Mabsadd L
.26286 .03401 1.46946 -2.02254 Mabsadd L
.26286 .03401 .91448 -0.29713 Mabsadd L
.26286 .03401 2.37764 .77254 Mabsadd L
.26286 .03401 .56518 .7779 Mabsadd L
.26286 .03401 0 2.5 Mabsadd L closepath F
.26381 .03401 0 2.5 Mabsadd m
.26381 .03401 -0.56518 .7779 Mabsadd L
.26381 .03401 -2.37764 .77254 Mabsadd L
.26381 .03401 -0.91448 -0.29713 Mabsadd L
.26381 .03401 -1.46946 -2.02254 Mabsadd L
.26381 .03401 0 -0.96154 Mabsadd L
.26381 .03401 1.46946 -2.02254 Mabsadd L
.26381 .03401 .91448 -0.29713 Mabsadd L
.26381 .03401 2.37764 .77254 Mabsadd L
.26381 .03401 .56518 .7779 Mabsadd L
.26381 .03401 0 2.5 Mabsadd L closepath F
.26476 .03402 0 2.5 Mabsadd m
.26476 .03402 -0.56518 .7779 Mabsadd L
.26476 .03402 -2.37764 .77254 Mabsadd L
.26476 .03402 -0.91448 -0.29713 Mabsadd L
.26476 .03402 -1.46946 -2.02254 Mabsadd L
.26476 .03402 0 -0.96154 Mabsadd L
.26476 .03402 1.46946 -2.02254 Mabsadd L
.26476 .03402 .91448 -0.29713 Mabsadd L
.26476 .03402 2.37764 .77254 Mabsadd L
.26476 .03402 .56518 .7779 Mabsadd L
.26476 .03402 0 2.5 Mabsadd L closepath F
.26571 .03402 0 2.5 Mabsadd m
.26571 .03402 -0.56518 .7779 Mabsadd L
.26571 .03402 -2.37764 .77254 Mabsadd L
.26571 .03402 -0.91448 -0.29713 Mabsadd L
.26571 .03402 -1.46946 -2.02254 Mabsadd L
.26571 .03402 0 -0.96154 Mabsadd L
.26571 .03402 1.46946 -2.02254 Mabsadd L
.26571 .03402 .91448 -0.29713 Mabsadd L
.26571 .03402 2.37764 .77254 Mabsadd L
.26571 .03402 .56518 .7779 Mabsadd L
.26571 .03402 0 2.5 Mabsadd L closepath F
.26667 .03401 0 2.5 Mabsadd m
.26667 .03401 -0.56518 .7779 Mabsadd L
.26667 .03401 -2.37764 .77254 Mabsadd L
.26667 .03401 -0.91448 -0.29713 Mabsadd L
.26667 .03401 -1.46946 -2.02254 Mabsadd L
.26667 .03401 0 -0.96154 Mabsadd L
.26667 .03401 1.46946 -2.02254 Mabsadd L
.26667 .03401 .91448 -0.29713 Mabsadd L
.26667 .03401 2.37764 .77254 Mabsadd L
.26667 .03401 .56518 .7779 Mabsadd L
.26667 .03401 0 2.5 Mabsadd L closepath F
.26762 .03401 0 2.5 Mabsadd m
.26762 .03401 -0.56518 .7779 Mabsadd L
.26762 .03401 -2.37764 .77254 Mabsadd L
.26762 .03401 -0.91448 -0.29713 Mabsadd L
.26762 .03401 -1.46946 -2.02254 Mabsadd L
.26762 .03401 0 -0.96154 Mabsadd L
.26762 .03401 1.46946 -2.02254 Mabsadd L
.26762 .03401 .91448 -0.29713 Mabsadd L
.26762 .03401 2.37764 .77254 Mabsadd L
.26762 .03401 .56518 .7779 Mabsadd L
.26762 .03401 0 2.5 Mabsadd L closepath F
.26857 .03401 0 2.5 Mabsadd m
.26857 .03401 -0.56518 .7779 Mabsadd L
.26857 .03401 -2.37764 .77254 Mabsadd L
.26857 .03401 -0.91448 -0.29713 Mabsadd L
.26857 .03401 -1.46946 -2.02254 Mabsadd L
.26857 .03401 0 -0.96154 Mabsadd L
.26857 .03401 1.46946 -2.02254 Mabsadd L
.26857 .03401 .91448 -0.29713 Mabsadd L
.26857 .03401 2.37764 .77254 Mabsadd L
.26857 .03401 .56518 .7779 Mabsadd L
.26857 .03401 0 2.5 Mabsadd L closepath F
.26952 .03401 0 2.5 Mabsadd m
.26952 .03401 -0.56518 .7779 Mabsadd L
.26952 .03401 -2.37764 .77254 Mabsadd L
.26952 .03401 -0.91448 -0.29713 Mabsadd L
.26952 .03401 -1.46946 -2.02254 Mabsadd L
.26952 .03401 0 -0.96154 Mabsadd L
.26952 .03401 1.46946 -2.02254 Mabsadd L
.26952 .03401 .91448 -0.29713 Mabsadd L
.26952 .03401 2.37764 .77254 Mabsadd L
.26952 .03401 .56518 .7779 Mabsadd L
.26952 .03401 0 2.5 Mabsadd L closepath F
.27048 .03401 0 2.5 Mabsadd m
.27048 .03401 -0.56518 .7779 Mabsadd L
.27048 .03401 -2.37764 .77254 Mabsadd L
.27048 .03401 -0.91448 -0.29713 Mabsadd L
.27048 .03401 -1.46946 -2.02254 Mabsadd L
.27048 .03401 0 -0.96154 Mabsadd L
.27048 .03401 1.46946 -2.02254 Mabsadd L
.27048 .03401 .91448 -0.29713 Mabsadd L
.27048 .03401 2.37764 .77254 Mabsadd L
.27048 .03401 .56518 .7779 Mabsadd L
.27048 .03401 0 2.5 Mabsadd L closepath F
.27143 .034 0 2.5 Mabsadd m
.27143 .034 -0.56518 .7779 Mabsadd L
.27143 .034 -2.37764 .77254 Mabsadd L
.27143 .034 -0.91448 -0.29713 Mabsadd L
.27143 .034 -1.46946 -2.02254 Mabsadd L
.27143 .034 0 -0.96154 Mabsadd L
.27143 .034 1.46946 -2.02254 Mabsadd L
.27143 .034 .91448 -0.29713 Mabsadd L
.27143 .034 2.37764 .77254 Mabsadd L
.27143 .034 .56518 .7779 Mabsadd L
.27143 .034 0 2.5 Mabsadd L closepath F
.27238 .034 0 2.5 Mabsadd m
.27238 .034 -0.56518 .7779 Mabsadd L
.27238 .034 -2.37764 .77254 Mabsadd L
.27238 .034 -0.91448 -0.29713 Mabsadd L
.27238 .034 -1.46946 -2.02254 Mabsadd L
.27238 .034 0 -0.96154 Mabsadd L
.27238 .034 1.46946 -2.02254 Mabsadd L
.27238 .034 .91448 -0.29713 Mabsadd L
.27238 .034 2.37764 .77254 Mabsadd L
.27238 .034 .56518 .7779 Mabsadd L
.27238 .034 0 2.5 Mabsadd L closepath F
.27333 .03399 0 2.5 Mabsadd m
.27333 .03399 -0.56518 .7779 Mabsadd L
.27333 .03399 -2.37764 .77254 Mabsadd L
.27333 .03399 -0.91448 -0.29713 Mabsadd L
.27333 .03399 -1.46946 -2.02254 Mabsadd L
.27333 .03399 0 -0.96154 Mabsadd L
.27333 .03399 1.46946 -2.02254 Mabsadd L
.27333 .03399 .91448 -0.29713 Mabsadd L
.27333 .03399 2.37764 .77254 Mabsadd L
.27333 .03399 .56518 .7779 Mabsadd L
.27333 .03399 0 2.5 Mabsadd L closepath F
.27429 .03397 0 2.5 Mabsadd m
.27429 .03397 -0.56518 .7779 Mabsadd L
.27429 .03397 -2.37764 .77254 Mabsadd L
.27429 .03397 -0.91448 -0.29713 Mabsadd L
.27429 .03397 -1.46946 -2.02254 Mabsadd L
.27429 .03397 0 -0.96154 Mabsadd L
.27429 .03397 1.46946 -2.02254 Mabsadd L
.27429 .03397 .91448 -0.29713 Mabsadd L
.27429 .03397 2.37764 .77254 Mabsadd L
.27429 .03397 .56518 .7779 Mabsadd L
.27429 .03397 0 2.5 Mabsadd L closepath F
.27524 .03397 0 2.5 Mabsadd m
.27524 .03397 -0.56518 .7779 Mabsadd L
.27524 .03397 -2.37764 .77254 Mabsadd L
.27524 .03397 -0.91448 -0.29713 Mabsadd L
.27524 .03397 -1.46946 -2.02254 Mabsadd L
.27524 .03397 0 -0.96154 Mabsadd L
.27524 .03397 1.46946 -2.02254 Mabsadd L
.27524 .03397 .91448 -0.29713 Mabsadd L
.27524 .03397 2.37764 .77254 Mabsadd L
.27524 .03397 .56518 .7779 Mabsadd L
.27524 .03397 0 2.5 Mabsadd L closepath F
.27619 .03394 0 2.5 Mabsadd m
.27619 .03394 -0.56518 .7779 Mabsadd L
.27619 .03394 -2.37764 .77254 Mabsadd L
.27619 .03394 -0.91448 -0.29713 Mabsadd L
.27619 .03394 -1.46946 -2.02254 Mabsadd L
.27619 .03394 0 -0.96154 Mabsadd L
.27619 .03394 1.46946 -2.02254 Mabsadd L
.27619 .03394 .91448 -0.29713 Mabsadd L
.27619 .03394 2.37764 .77254 Mabsadd L
.27619 .03394 .56518 .7779 Mabsadd L
.27619 .03394 0 2.5 Mabsadd L closepath F
.27714 .03385 0 2.5 Mabsadd m
.27714 .03385 -0.56518 .7779 Mabsadd L
.27714 .03385 -2.37764 .77254 Mabsadd L
.27714 .03385 -0.91448 -0.29713 Mabsadd L
.27714 .03385 -1.46946 -2.02254 Mabsadd L
.27714 .03385 0 -0.96154 Mabsadd L
.27714 .03385 1.46946 -2.02254 Mabsadd L
.27714 .03385 .91448 -0.29713 Mabsadd L
.27714 .03385 2.37764 .77254 Mabsadd L
.27714 .03385 .56518 .7779 Mabsadd L
.27714 .03385 0 2.5 Mabsadd L closepath F
.2781 .03383 0 2.5 Mabsadd m
.2781 .03383 -0.56518 .7779 Mabsadd L
.2781 .03383 -2.37764 .77254 Mabsadd L
.2781 .03383 -0.91448 -0.29713 Mabsadd L
.2781 .03383 -1.46946 -2.02254 Mabsadd L
.2781 .03383 0 -0.96154 Mabsadd L
.2781 .03383 1.46946 -2.02254 Mabsadd L
.2781 .03383 .91448 -0.29713 Mabsadd L
.2781 .03383 2.37764 .77254 Mabsadd L
.2781 .03383 .56518 .7779 Mabsadd L
.2781 .03383 0 2.5 Mabsadd L closepath F
.27905 .03366 0 2.5 Mabsadd m
.27905 .03366 -0.56518 .7779 Mabsadd L
.27905 .03366 -2.37764 .77254 Mabsadd L
.27905 .03366 -0.91448 -0.29713 Mabsadd L
.27905 .03366 -1.46946 -2.02254 Mabsadd L
.27905 .03366 0 -0.96154 Mabsadd L
.27905 .03366 1.46946 -2.02254 Mabsadd L
.27905 .03366 .91448 -0.29713 Mabsadd L
.27905 .03366 2.37764 .77254 Mabsadd L
.27905 .03366 .56518 .7779 Mabsadd L
.27905 .03366 0 2.5 Mabsadd L closepath F
.28 .03292 0 2.5 Mabsadd m
.28 .03292 -0.56518 .7779 Mabsadd L
.28 .03292 -2.37764 .77254 Mabsadd L
.28 .03292 -0.91448 -0.29713 Mabsadd L
.28 .03292 -1.46946 -2.02254 Mabsadd L
.28 .03292 0 -0.96154 Mabsadd L
.28 .03292 1.46946 -2.02254 Mabsadd L
.28 .03292 .91448 -0.29713 Mabsadd L
.28 .03292 2.37764 .77254 Mabsadd L
.28 .03292 .56518 .7779 Mabsadd L
.28 .03292 0 2.5 Mabsadd L closepath F
.28095 .03272 0 2.5 Mabsadd m
.28095 .03272 -0.56518 .7779 Mabsadd L
.28095 .03272 -2.37764 .77254 Mabsadd L
.28095 .03272 -0.91448 -0.29713 Mabsadd L
.28095 .03272 -1.46946 -2.02254 Mabsadd L
.28095 .03272 0 -0.96154 Mabsadd L
.28095 .03272 1.46946 -2.02254 Mabsadd L
.28095 .03272 .91448 -0.29713 Mabsadd L
.28095 .03272 2.37764 .77254 Mabsadd L
.28095 .03272 .56518 .7779 Mabsadd L
.28095 .03272 0 2.5 Mabsadd L closepath F
.2819 .0307 0 2.5 Mabsadd m
.2819 .0307 -0.56518 .7779 Mabsadd L
.2819 .0307 -2.37764 .77254 Mabsadd L
.2819 .0307 -0.91448 -0.29713 Mabsadd L
.2819 .0307 -1.46946 -2.02254 Mabsadd L
.2819 .0307 0 -0.96154 Mabsadd L
.2819 .0307 1.46946 -2.02254 Mabsadd L
.2819 .0307 .91448 -0.29713 Mabsadd L
.2819 .0307 2.37764 .77254 Mabsadd L
.2819 .0307 .56518 .7779 Mabsadd L
.2819 .0307 0 2.5 Mabsadd L closepath F
.28286 .0294 0 2.5 Mabsadd m
.28286 .0294 -0.56518 .7779 Mabsadd L
.28286 .0294 -2.37764 .77254 Mabsadd L
.28286 .0294 -0.91448 -0.29713 Mabsadd L
.28286 .0294 -1.46946 -2.02254 Mabsadd L
.28286 .0294 0 -0.96154 Mabsadd L
.28286 .0294 1.46946 -2.02254 Mabsadd L
.28286 .0294 .91448 -0.29713 Mabsadd L
.28286 .0294 2.37764 .77254 Mabsadd L
.28286 .0294 .56518 .7779 Mabsadd L
.28286 .0294 0 2.5 Mabsadd L closepath F
.28381 .03337 0 2.5 Mabsadd m
.28381 .03337 -0.56518 .7779 Mabsadd L
.28381 .03337 -2.37764 .77254 Mabsadd L
.28381 .03337 -0.91448 -0.29713 Mabsadd L
.28381 .03337 -1.46946 -2.02254 Mabsadd L
.28381 .03337 0 -0.96154 Mabsadd L
.28381 .03337 1.46946 -2.02254 Mabsadd L
.28381 .03337 .91448 -0.29713 Mabsadd L
.28381 .03337 2.37764 .77254 Mabsadd L
.28381 .03337 .56518 .7779 Mabsadd L
.28381 .03337 0 2.5 Mabsadd L closepath F
.28476 .04602 0 2.5 Mabsadd m
.28476 .04602 -0.56518 .7779 Mabsadd L
.28476 .04602 -2.37764 .77254 Mabsadd L
.28476 .04602 -0.91448 -0.29713 Mabsadd L
.28476 .04602 -1.46946 -2.02254 Mabsadd L
.28476 .04602 0 -0.96154 Mabsadd L
.28476 .04602 1.46946 -2.02254 Mabsadd L
.28476 .04602 .91448 -0.29713 Mabsadd L
.28476 .04602 2.37764 .77254 Mabsadd L
.28476 .04602 .56518 .7779 Mabsadd L
.28476 .04602 0 2.5 Mabsadd L closepath F
.28571 .09233 0 2.5 Mabsadd m
.28571 .09233 -0.56518 .7779 Mabsadd L
.28571 .09233 -2.37764 .77254 Mabsadd L
.28571 .09233 -0.91448 -0.29713 Mabsadd L
.28571 .09233 -1.46946 -2.02254 Mabsadd L
.28571 .09233 0 -0.96154 Mabsadd L
.28571 .09233 1.46946 -2.02254 Mabsadd L
.28571 .09233 .91448 -0.29713 Mabsadd L
.28571 .09233 2.37764 .77254 Mabsadd L
.28571 .09233 .56518 .7779 Mabsadd L
.28571 .09233 0 2.5 Mabsadd L closepath F
.28667 .0923 0 2.5 Mabsadd m
.28667 .0923 -0.56518 .7779 Mabsadd L
.28667 .0923 -2.37764 .77254 Mabsadd L
.28667 .0923 -0.91448 -0.29713 Mabsadd L
.28667 .0923 -1.46946 -2.02254 Mabsadd L
.28667 .0923 0 -0.96154 Mabsadd L
.28667 .0923 1.46946 -2.02254 Mabsadd L
.28667 .0923 .91448 -0.29713 Mabsadd L
.28667 .0923 2.37764 .77254 Mabsadd L
.28667 .0923 .56518 .7779 Mabsadd L
.28667 .0923 0 2.5 Mabsadd L closepath F
.28762 .09001 0 2.5 Mabsadd m
.28762 .09001 -0.56518 .7779 Mabsadd L
.28762 .09001 -2.37764 .77254 Mabsadd L
.28762 .09001 -0.91448 -0.29713 Mabsadd L
.28762 .09001 -1.46946 -2.02254 Mabsadd L
.28762 .09001 0 -0.96154 Mabsadd L
.28762 .09001 1.46946 -2.02254 Mabsadd L
.28762 .09001 .91448 -0.29713 Mabsadd L
.28762 .09001 2.37764 .77254 Mabsadd L
.28762 .09001 .56518 .7779 Mabsadd L
.28762 .09001 0 2.5 Mabsadd L closepath F
.28857 .08565 0 2.5 Mabsadd m
.28857 .08565 -0.56518 .7779 Mabsadd L
.28857 .08565 -2.37764 .77254 Mabsadd L
.28857 .08565 -0.91448 -0.29713 Mabsadd L
.28857 .08565 -1.46946 -2.02254 Mabsadd L
.28857 .08565 0 -0.96154 Mabsadd L
.28857 .08565 1.46946 -2.02254 Mabsadd L
.28857 .08565 .91448 -0.29713 Mabsadd L
.28857 .08565 2.37764 .77254 Mabsadd L
.28857 .08565 .56518 .7779 Mabsadd L
.28857 .08565 0 2.5 Mabsadd L closepath F
.28952 .08547 0 2.5 Mabsadd m
.28952 .08547 -0.56518 .7779 Mabsadd L
.28952 .08547 -2.37764 .77254 Mabsadd L
.28952 .08547 -0.91448 -0.29713 Mabsadd L
.28952 .08547 -1.46946 -2.02254 Mabsadd L
.28952 .08547 0 -0.96154 Mabsadd L
.28952 .08547 1.46946 -2.02254 Mabsadd L
.28952 .08547 .91448 -0.29713 Mabsadd L
.28952 .08547 2.37764 .77254 Mabsadd L
.28952 .08547 .56518 .7779 Mabsadd L
.28952 .08547 0 2.5 Mabsadd L closepath F
.29048 .08503 0 2.5 Mabsadd m
.29048 .08503 -0.56518 .7779 Mabsadd L
.29048 .08503 -2.37764 .77254 Mabsadd L
.29048 .08503 -0.91448 -0.29713 Mabsadd L
.29048 .08503 -1.46946 -2.02254 Mabsadd L
.29048 .08503 0 -0.96154 Mabsadd L
.29048 .08503 1.46946 -2.02254 Mabsadd L
.29048 .08503 .91448 -0.29713 Mabsadd L
.29048 .08503 2.37764 .77254 Mabsadd L
.29048 .08503 .56518 .7779 Mabsadd L
.29048 .08503 0 2.5 Mabsadd L closepath F
.29143 .08189 0 2.5 Mabsadd m
.29143 .08189 -0.56518 .7779 Mabsadd L
.29143 .08189 -2.37764 .77254 Mabsadd L
.29143 .08189 -0.91448 -0.29713 Mabsadd L
.29143 .08189 -1.46946 -2.02254 Mabsadd L
.29143 .08189 0 -0.96154 Mabsadd L
.29143 .08189 1.46946 -2.02254 Mabsadd L
.29143 .08189 .91448 -0.29713 Mabsadd L
.29143 .08189 2.37764 .77254 Mabsadd L
.29143 .08189 .56518 .7779 Mabsadd L
.29143 .08189 0 2.5 Mabsadd L closepath F
.29238 .0792 0 2.5 Mabsadd m
.29238 .0792 -0.56518 .7779 Mabsadd L
.29238 .0792 -2.37764 .77254 Mabsadd L
.29238 .0792 -0.91448 -0.29713 Mabsadd L
.29238 .0792 -1.46946 -2.02254 Mabsadd L
.29238 .0792 0 -0.96154 Mabsadd L
.29238 .0792 1.46946 -2.02254 Mabsadd L
.29238 .0792 .91448 -0.29713 Mabsadd L
.29238 .0792 2.37764 .77254 Mabsadd L
.29238 .0792 .56518 .7779 Mabsadd L
.29238 .0792 0 2.5 Mabsadd L closepath F
.29333 .07944 0 2.5 Mabsadd m
.29333 .07944 -0.56518 .7779 Mabsadd L
.29333 .07944 -2.37764 .77254 Mabsadd L
.29333 .07944 -0.91448 -0.29713 Mabsadd L
.29333 .07944 -1.46946 -2.02254 Mabsadd L
.29333 .07944 0 -0.96154 Mabsadd L
.29333 .07944 1.46946 -2.02254 Mabsadd L
.29333 .07944 .91448 -0.29713 Mabsadd L
.29333 .07944 2.37764 .77254 Mabsadd L
.29333 .07944 .56518 .7779 Mabsadd L
.29333 .07944 0 2.5 Mabsadd L closepath F
.29429 .02747 0 2.5 Mabsadd m
.29429 .02747 -0.56518 .7779 Mabsadd L
.29429 .02747 -2.37764 .77254 Mabsadd L
.29429 .02747 -0.91448 -0.29713 Mabsadd L
.29429 .02747 -1.46946 -2.02254 Mabsadd L
.29429 .02747 0 -0.96154 Mabsadd L
.29429 .02747 1.46946 -2.02254 Mabsadd L
.29429 .02747 .91448 -0.29713 Mabsadd L
.29429 .02747 2.37764 .77254 Mabsadd L
.29429 .02747 .56518 .7779 Mabsadd L
.29429 .02747 0 2.5 Mabsadd L closepath F
.29524 .03641 0 2.5 Mabsadd m
.29524 .03641 -0.56518 .7779 Mabsadd L
.29524 .03641 -2.37764 .77254 Mabsadd L
.29524 .03641 -0.91448 -0.29713 Mabsadd L
.29524 .03641 -1.46946 -2.02254 Mabsadd L
.29524 .03641 0 -0.96154 Mabsadd L
.29524 .03641 1.46946 -2.02254 Mabsadd L
.29524 .03641 .91448 -0.29713 Mabsadd L
.29524 .03641 2.37764 .77254 Mabsadd L
.29524 .03641 .56518 .7779 Mabsadd L
.29524 .03641 0 2.5 Mabsadd L closepath F
.29619 .06683 0 2.5 Mabsadd m
.29619 .06683 -0.56518 .7779 Mabsadd L
.29619 .06683 -2.37764 .77254 Mabsadd L
.29619 .06683 -0.91448 -0.29713 Mabsadd L
.29619 .06683 -1.46946 -2.02254 Mabsadd L
.29619 .06683 0 -0.96154 Mabsadd L
.29619 .06683 1.46946 -2.02254 Mabsadd L
.29619 .06683 .91448 -0.29713 Mabsadd L
.29619 .06683 2.37764 .77254 Mabsadd L
.29619 .06683 .56518 .7779 Mabsadd L
.29619 .06683 0 2.5 Mabsadd L closepath F
.29714 .06689 0 2.5 Mabsadd m
.29714 .06689 -0.56518 .7779 Mabsadd L
.29714 .06689 -2.37764 .77254 Mabsadd L
.29714 .06689 -0.91448 -0.29713 Mabsadd L
.29714 .06689 -1.46946 -2.02254 Mabsadd L
.29714 .06689 0 -0.96154 Mabsadd L
.29714 .06689 1.46946 -2.02254 Mabsadd L
.29714 .06689 .91448 -0.29713 Mabsadd L
.29714 .06689 2.37764 .77254 Mabsadd L
.29714 .06689 .56518 .7779 Mabsadd L
.29714 .06689 0 2.5 Mabsadd L closepath F
.2981 .06707 0 2.5 Mabsadd m
.2981 .06707 -0.56518 .7779 Mabsadd L
.2981 .06707 -2.37764 .77254 Mabsadd L
.2981 .06707 -0.91448 -0.29713 Mabsadd L
.2981 .06707 -1.46946 -2.02254 Mabsadd L
.2981 .06707 0 -0.96154 Mabsadd L
.2981 .06707 1.46946 -2.02254 Mabsadd L
.2981 .06707 .91448 -0.29713 Mabsadd L
.2981 .06707 2.37764 .77254 Mabsadd L
.2981 .06707 .56518 .7779 Mabsadd L
.2981 .06707 0 2.5 Mabsadd L closepath F
.29905 .06715 0 2.5 Mabsadd m
.29905 .06715 -0.56518 .7779 Mabsadd L
.29905 .06715 -2.37764 .77254 Mabsadd L
.29905 .06715 -0.91448 -0.29713 Mabsadd L
.29905 .06715 -1.46946 -2.02254 Mabsadd L
.29905 .06715 0 -0.96154 Mabsadd L
.29905 .06715 1.46946 -2.02254 Mabsadd L
.29905 .06715 .91448 -0.29713 Mabsadd L
.29905 .06715 2.37764 .77254 Mabsadd L
.29905 .06715 .56518 .7779 Mabsadd L
.29905 .06715 0 2.5 Mabsadd L closepath F
.3 .06708 0 2.5 Mabsadd m
.3 .06708 -0.56518 .7779 Mabsadd L
.3 .06708 -2.37764 .77254 Mabsadd L
.3 .06708 -0.91448 -0.29713 Mabsadd L
.3 .06708 -1.46946 -2.02254 Mabsadd L
.3 .06708 0 -0.96154 Mabsadd L
.3 .06708 1.46946 -2.02254 Mabsadd L
.3 .06708 .91448 -0.29713 Mabsadd L
.3 .06708 2.37764 .77254 Mabsadd L
.3 .06708 .56518 .7779 Mabsadd L
.3 .06708 0 2.5 Mabsadd L closepath F
.30095 .06692 0 2.5 Mabsadd m
.30095 .06692 -0.56518 .7779 Mabsadd L
.30095 .06692 -2.37764 .77254 Mabsadd L
.30095 .06692 -0.91448 -0.29713 Mabsadd L
.30095 .06692 -1.46946 -2.02254 Mabsadd L
.30095 .06692 0 -0.96154 Mabsadd L
.30095 .06692 1.46946 -2.02254 Mabsadd L
.30095 .06692 .91448 -0.29713 Mabsadd L
.30095 .06692 2.37764 .77254 Mabsadd L
.30095 .06692 .56518 .7779 Mabsadd L
.30095 .06692 0 2.5 Mabsadd L closepath F
.3019 .06756 0 2.5 Mabsadd m
.3019 .06756 -0.56518 .7779 Mabsadd L
.3019 .06756 -2.37764 .77254 Mabsadd L
.3019 .06756 -0.91448 -0.29713 Mabsadd L
.3019 .06756 -1.46946 -2.02254 Mabsadd L
.3019 .06756 0 -0.96154 Mabsadd L
.3019 .06756 1.46946 -2.02254 Mabsadd L
.3019 .06756 .91448 -0.29713 Mabsadd L
.3019 .06756 2.37764 .77254 Mabsadd L
.3019 .06756 .56518 .7779 Mabsadd L
.3019 .06756 0 2.5 Mabsadd L closepath F
.30286 .06952 0 2.5 Mabsadd m
.30286 .06952 -0.56518 .7779 Mabsadd L
.30286 .06952 -2.37764 .77254 Mabsadd L
.30286 .06952 -0.91448 -0.29713 Mabsadd L
.30286 .06952 -1.46946 -2.02254 Mabsadd L
.30286 .06952 0 -0.96154 Mabsadd L
.30286 .06952 1.46946 -2.02254 Mabsadd L
.30286 .06952 .91448 -0.29713 Mabsadd L
.30286 .06952 2.37764 .77254 Mabsadd L
.30286 .06952 .56518 .7779 Mabsadd L
.30286 .06952 0 2.5 Mabsadd L closepath F
.30381 .06963 0 2.5 Mabsadd m
.30381 .06963 -0.56518 .7779 Mabsadd L
.30381 .06963 -2.37764 .77254 Mabsadd L
.30381 .06963 -0.91448 -0.29713 Mabsadd L
.30381 .06963 -1.46946 -2.02254 Mabsadd L
.30381 .06963 0 -0.96154 Mabsadd L
.30381 .06963 1.46946 -2.02254 Mabsadd L
.30381 .06963 .91448 -0.29713 Mabsadd L
.30381 .06963 2.37764 .77254 Mabsadd L
.30381 .06963 .56518 .7779 Mabsadd L
.30381 .06963 0 2.5 Mabsadd L closepath F
.30476 .06999 0 2.5 Mabsadd m
.30476 .06999 -0.56518 .7779 Mabsadd L
.30476 .06999 -2.37764 .77254 Mabsadd L
.30476 .06999 -0.91448 -0.29713 Mabsadd L
.30476 .06999 -1.46946 -2.02254 Mabsadd L
.30476 .06999 0 -0.96154 Mabsadd L
.30476 .06999 1.46946 -2.02254 Mabsadd L
.30476 .06999 .91448 -0.29713 Mabsadd L
.30476 .06999 2.37764 .77254 Mabsadd L
.30476 .06999 .56518 .7779 Mabsadd L
.30476 .06999 0 2.5 Mabsadd L closepath F
.30571 .07002 0 2.5 Mabsadd m
.30571 .07002 -0.56518 .7779 Mabsadd L
.30571 .07002 -2.37764 .77254 Mabsadd L
.30571 .07002 -0.91448 -0.29713 Mabsadd L
.30571 .07002 -1.46946 -2.02254 Mabsadd L
.30571 .07002 0 -0.96154 Mabsadd L
.30571 .07002 1.46946 -2.02254 Mabsadd L
.30571 .07002 .91448 -0.29713 Mabsadd L
.30571 .07002 2.37764 .77254 Mabsadd L
.30571 .07002 .56518 .7779 Mabsadd L
.30571 .07002 0 2.5 Mabsadd L closepath F
.30667 .07011 0 2.5 Mabsadd m
.30667 .07011 -0.56518 .7779 Mabsadd L
.30667 .07011 -2.37764 .77254 Mabsadd L
.30667 .07011 -0.91448 -0.29713 Mabsadd L
.30667 .07011 -1.46946 -2.02254 Mabsadd L
.30667 .07011 0 -0.96154 Mabsadd L
.30667 .07011 1.46946 -2.02254 Mabsadd L
.30667 .07011 .91448 -0.29713 Mabsadd L
.30667 .07011 2.37764 .77254 Mabsadd L
.30667 .07011 .56518 .7779 Mabsadd L
.30667 .07011 0 2.5 Mabsadd L closepath F
.30762 .07024 0 2.5 Mabsadd m
.30762 .07024 -0.56518 .7779 Mabsadd L
.30762 .07024 -2.37764 .77254 Mabsadd L
.30762 .07024 -0.91448 -0.29713 Mabsadd L
.30762 .07024 -1.46946 -2.02254 Mabsadd L
.30762 .07024 0 -0.96154 Mabsadd L
.30762 .07024 1.46946 -2.02254 Mabsadd L
.30762 .07024 .91448 -0.29713 Mabsadd L
.30762 .07024 2.37764 .77254 Mabsadd L
.30762 .07024 .56518 .7779 Mabsadd L
.30762 .07024 0 2.5 Mabsadd L closepath F
.30857 .07025 0 2.5 Mabsadd m
.30857 .07025 -0.56518 .7779 Mabsadd L
.30857 .07025 -2.37764 .77254 Mabsadd L
.30857 .07025 -0.91448 -0.29713 Mabsadd L
.30857 .07025 -1.46946 -2.02254 Mabsadd L
.30857 .07025 0 -0.96154 Mabsadd L
.30857 .07025 1.46946 -2.02254 Mabsadd L
.30857 .07025 .91448 -0.29713 Mabsadd L
.30857 .07025 2.37764 .77254 Mabsadd L
.30857 .07025 .56518 .7779 Mabsadd L
.30857 .07025 0 2.5 Mabsadd L closepath F
.30952 .07026 0 2.5 Mabsadd m
.30952 .07026 -0.56518 .7779 Mabsadd L
.30952 .07026 -2.37764 .77254 Mabsadd L
.30952 .07026 -0.91448 -0.29713 Mabsadd L
.30952 .07026 -1.46946 -2.02254 Mabsadd L
.30952 .07026 0 -0.96154 Mabsadd L
.30952 .07026 1.46946 -2.02254 Mabsadd L
.30952 .07026 .91448 -0.29713 Mabsadd L
.30952 .07026 2.37764 .77254 Mabsadd L
.30952 .07026 .56518 .7779 Mabsadd L
.30952 .07026 0 2.5 Mabsadd L closepath F
.31048 .07029 0 2.5 Mabsadd m
.31048 .07029 -0.56518 .7779 Mabsadd L
.31048 .07029 -2.37764 .77254 Mabsadd L
.31048 .07029 -0.91448 -0.29713 Mabsadd L
.31048 .07029 -1.46946 -2.02254 Mabsadd L
.31048 .07029 0 -0.96154 Mabsadd L
.31048 .07029 1.46946 -2.02254 Mabsadd L
.31048 .07029 .91448 -0.29713 Mabsadd L
.31048 .07029 2.37764 .77254 Mabsadd L
.31048 .07029 .56518 .7779 Mabsadd L
.31048 .07029 0 2.5 Mabsadd L closepath F
.31143 .07029 0 2.5 Mabsadd m
.31143 .07029 -0.56518 .7779 Mabsadd L
.31143 .07029 -2.37764 .77254 Mabsadd L
.31143 .07029 -0.91448 -0.29713 Mabsadd L
.31143 .07029 -1.46946 -2.02254 Mabsadd L
.31143 .07029 0 -0.96154 Mabsadd L
.31143 .07029 1.46946 -2.02254 Mabsadd L
.31143 .07029 .91448 -0.29713 Mabsadd L
.31143 .07029 2.37764 .77254 Mabsadd L
.31143 .07029 .56518 .7779 Mabsadd L
.31143 .07029 0 2.5 Mabsadd L closepath F
.31238 .07029 0 2.5 Mabsadd m
.31238 .07029 -0.56518 .7779 Mabsadd L
.31238 .07029 -2.37764 .77254 Mabsadd L
.31238 .07029 -0.91448 -0.29713 Mabsadd L
.31238 .07029 -1.46946 -2.02254 Mabsadd L
.31238 .07029 0 -0.96154 Mabsadd L
.31238 .07029 1.46946 -2.02254 Mabsadd L
.31238 .07029 .91448 -0.29713 Mabsadd L
.31238 .07029 2.37764 .77254 Mabsadd L
.31238 .07029 .56518 .7779 Mabsadd L
.31238 .07029 0 2.5 Mabsadd L closepath F
.31333 .0703 0 2.5 Mabsadd m
.31333 .0703 -0.56518 .7779 Mabsadd L
.31333 .0703 -2.37764 .77254 Mabsadd L
.31333 .0703 -0.91448 -0.29713 Mabsadd L
.31333 .0703 -1.46946 -2.02254 Mabsadd L
.31333 .0703 0 -0.96154 Mabsadd L
.31333 .0703 1.46946 -2.02254 Mabsadd L
.31333 .0703 .91448 -0.29713 Mabsadd L
.31333 .0703 2.37764 .77254 Mabsadd L
.31333 .0703 .56518 .7779 Mabsadd L
.31333 .0703 0 2.5 Mabsadd L closepath F
.31429 .0703 0 2.5 Mabsadd m
.31429 .0703 -0.56518 .7779 Mabsadd L
.31429 .0703 -2.37764 .77254 Mabsadd L
.31429 .0703 -0.91448 -0.29713 Mabsadd L
.31429 .0703 -1.46946 -2.02254 Mabsadd L
.31429 .0703 0 -0.96154 Mabsadd L
.31429 .0703 1.46946 -2.02254 Mabsadd L
.31429 .0703 .91448 -0.29713 Mabsadd L
.31429 .0703 2.37764 .77254 Mabsadd L
.31429 .0703 .56518 .7779 Mabsadd L
.31429 .0703 0 2.5 Mabsadd L closepath F
.31524 .0703 0 2.5 Mabsadd m
.31524 .0703 -0.56518 .7779 Mabsadd L
.31524 .0703 -2.37764 .77254 Mabsadd L
.31524 .0703 -0.91448 -0.29713 Mabsadd L
.31524 .0703 -1.46946 -2.02254 Mabsadd L
.31524 .0703 0 -0.96154 Mabsadd L
.31524 .0703 1.46946 -2.02254 Mabsadd L
.31524 .0703 .91448 -0.29713 Mabsadd L
.31524 .0703 2.37764 .77254 Mabsadd L
.31524 .0703 .56518 .7779 Mabsadd L
.31524 .0703 0 2.5 Mabsadd L closepath F
.31619 .0703 0 2.5 Mabsadd m
.31619 .0703 -0.56518 .7779 Mabsadd L
.31619 .0703 -2.37764 .77254 Mabsadd L
.31619 .0703 -0.91448 -0.29713 Mabsadd L
.31619 .0703 -1.46946 -2.02254 Mabsadd L
.31619 .0703 0 -0.96154 Mabsadd L
.31619 .0703 1.46946 -2.02254 Mabsadd L
.31619 .0703 .91448 -0.29713 Mabsadd L
.31619 .0703 2.37764 .77254 Mabsadd L
.31619 .0703 .56518 .7779 Mabsadd L
.31619 .0703 0 2.5 Mabsadd L closepath F
.31714 .0703 0 2.5 Mabsadd m
.31714 .0703 -0.56518 .7779 Mabsadd L
.31714 .0703 -2.37764 .77254 Mabsadd L
.31714 .0703 -0.91448 -0.29713 Mabsadd L
.31714 .0703 -1.46946 -2.02254 Mabsadd L
.31714 .0703 0 -0.96154 Mabsadd L
.31714 .0703 1.46946 -2.02254 Mabsadd L
.31714 .0703 .91448 -0.29713 Mabsadd L
.31714 .0703 2.37764 .77254 Mabsadd L
.31714 .0703 .56518 .7779 Mabsadd L
.31714 .0703 0 2.5 Mabsadd L closepath F
.3181 .0703 0 2.5 Mabsadd m
.3181 .0703 -0.56518 .7779 Mabsadd L
.3181 .0703 -2.37764 .77254 Mabsadd L
.3181 .0703 -0.91448 -0.29713 Mabsadd L
.3181 .0703 -1.46946 -2.02254 Mabsadd L
.3181 .0703 0 -0.96154 Mabsadd L
.3181 .0703 1.46946 -2.02254 Mabsadd L
.3181 .0703 .91448 -0.29713 Mabsadd L
.3181 .0703 2.37764 .77254 Mabsadd L
.3181 .0703 .56518 .7779 Mabsadd L
.3181 .0703 0 2.5 Mabsadd L closepath F
.31905 .0703 0 2.5 Mabsadd m
.31905 .0703 -0.56518 .7779 Mabsadd L
.31905 .0703 -2.37764 .77254 Mabsadd L
.31905 .0703 -0.91448 -0.29713 Mabsadd L
.31905 .0703 -1.46946 -2.02254 Mabsadd L
.31905 .0703 0 -0.96154 Mabsadd L
.31905 .0703 1.46946 -2.02254 Mabsadd L
.31905 .0703 .91448 -0.29713 Mabsadd L
.31905 .0703 2.37764 .77254 Mabsadd L
.31905 .0703 .56518 .7779 Mabsadd L
.31905 .0703 0 2.5 Mabsadd L closepath F
.32 .0703 0 2.5 Mabsadd m
.32 .0703 -0.56518 .7779 Mabsadd L
.32 .0703 -2.37764 .77254 Mabsadd L
.32 .0703 -0.91448 -0.29713 Mabsadd L
.32 .0703 -1.46946 -2.02254 Mabsadd L
.32 .0703 0 -0.96154 Mabsadd L
.32 .0703 1.46946 -2.02254 Mabsadd L
.32 .0703 .91448 -0.29713 Mabsadd L
.32 .0703 2.37764 .77254 Mabsadd L
.32 .0703 .56518 .7779 Mabsadd L
.32 .0703 0 2.5 Mabsadd L closepath F
.32095 .07029 0 2.5 Mabsadd m
.32095 .07029 -0.56518 .7779 Mabsadd L
.32095 .07029 -2.37764 .77254 Mabsadd L
.32095 .07029 -0.91448 -0.29713 Mabsadd L
.32095 .07029 -1.46946 -2.02254 Mabsadd L
.32095 .07029 0 -0.96154 Mabsadd L
.32095 .07029 1.46946 -2.02254 Mabsadd L
.32095 .07029 .91448 -0.29713 Mabsadd L
.32095 .07029 2.37764 .77254 Mabsadd L
.32095 .07029 .56518 .7779 Mabsadd L
.32095 .07029 0 2.5 Mabsadd L closepath F
.3219 .07029 0 2.5 Mabsadd m
.3219 .07029 -0.56518 .7779 Mabsadd L
.3219 .07029 -2.37764 .77254 Mabsadd L
.3219 .07029 -0.91448 -0.29713 Mabsadd L
.3219 .07029 -1.46946 -2.02254 Mabsadd L
.3219 .07029 0 -0.96154 Mabsadd L
.3219 .07029 1.46946 -2.02254 Mabsadd L
.3219 .07029 .91448 -0.29713 Mabsadd L
.3219 .07029 2.37764 .77254 Mabsadd L
.3219 .07029 .56518 .7779 Mabsadd L
.3219 .07029 0 2.5 Mabsadd L closepath F
.32286 .07029 0 2.5 Mabsadd m
.32286 .07029 -0.56518 .7779 Mabsadd L
.32286 .07029 -2.37764 .77254 Mabsadd L
.32286 .07029 -0.91448 -0.29713 Mabsadd L
.32286 .07029 -1.46946 -2.02254 Mabsadd L
.32286 .07029 0 -0.96154 Mabsadd L
.32286 .07029 1.46946 -2.02254 Mabsadd L
.32286 .07029 .91448 -0.29713 Mabsadd L
.32286 .07029 2.37764 .77254 Mabsadd L
.32286 .07029 .56518 .7779 Mabsadd L
.32286 .07029 0 2.5 Mabsadd L closepath F
.32381 .07029 0 2.5 Mabsadd m
.32381 .07029 -0.56518 .7779 Mabsadd L
.32381 .07029 -2.37764 .77254 Mabsadd L
.32381 .07029 -0.91448 -0.29713 Mabsadd L
.32381 .07029 -1.46946 -2.02254 Mabsadd L
.32381 .07029 0 -0.96154 Mabsadd L
.32381 .07029 1.46946 -2.02254 Mabsadd L
.32381 .07029 .91448 -0.29713 Mabsadd L
.32381 .07029 2.37764 .77254 Mabsadd L
.32381 .07029 .56518 .7779 Mabsadd L
.32381 .07029 0 2.5 Mabsadd L closepath F
.32476 .07028 0 2.5 Mabsadd m
.32476 .07028 -0.56518 .7779 Mabsadd L
.32476 .07028 -2.37764 .77254 Mabsadd L
.32476 .07028 -0.91448 -0.29713 Mabsadd L
.32476 .07028 -1.46946 -2.02254 Mabsadd L
.32476 .07028 0 -0.96154 Mabsadd L
.32476 .07028 1.46946 -2.02254 Mabsadd L
.32476 .07028 .91448 -0.29713 Mabsadd L
.32476 .07028 2.37764 .77254 Mabsadd L
.32476 .07028 .56518 .7779 Mabsadd L
.32476 .07028 0 2.5 Mabsadd L closepath F
.32571 .07028 0 2.5 Mabsadd m
.32571 .07028 -0.56518 .7779 Mabsadd L
.32571 .07028 -2.37764 .77254 Mabsadd L
.32571 .07028 -0.91448 -0.29713 Mabsadd L
.32571 .07028 -1.46946 -2.02254 Mabsadd L
.32571 .07028 0 -0.96154 Mabsadd L
.32571 .07028 1.46946 -2.02254 Mabsadd L
.32571 .07028 .91448 -0.29713 Mabsadd L
.32571 .07028 2.37764 .77254 Mabsadd L
.32571 .07028 .56518 .7779 Mabsadd L
.32571 .07028 0 2.5 Mabsadd L closepath F
.32667 .07028 0 2.5 Mabsadd m
.32667 .07028 -0.56518 .7779 Mabsadd L
.32667 .07028 -2.37764 .77254 Mabsadd L
.32667 .07028 -0.91448 -0.29713 Mabsadd L
.32667 .07028 -1.46946 -2.02254 Mabsadd L
.32667 .07028 0 -0.96154 Mabsadd L
.32667 .07028 1.46946 -2.02254 Mabsadd L
.32667 .07028 .91448 -0.29713 Mabsadd L
.32667 .07028 2.37764 .77254 Mabsadd L
.32667 .07028 .56518 .7779 Mabsadd L
.32667 .07028 0 2.5 Mabsadd L closepath F
.32762 .07026 0 2.5 Mabsadd m
.32762 .07026 -0.56518 .7779 Mabsadd L
.32762 .07026 -2.37764 .77254 Mabsadd L
.32762 .07026 -0.91448 -0.29713 Mabsadd L
.32762 .07026 -1.46946 -2.02254 Mabsadd L
.32762 .07026 0 -0.96154 Mabsadd L
.32762 .07026 1.46946 -2.02254 Mabsadd L
.32762 .07026 .91448 -0.29713 Mabsadd L
.32762 .07026 2.37764 .77254 Mabsadd L
.32762 .07026 .56518 .7779 Mabsadd L
.32762 .07026 0 2.5 Mabsadd L closepath F
.32857 .07026 0 2.5 Mabsadd m
.32857 .07026 -0.56518 .7779 Mabsadd L
.32857 .07026 -2.37764 .77254 Mabsadd L
.32857 .07026 -0.91448 -0.29713 Mabsadd L
.32857 .07026 -1.46946 -2.02254 Mabsadd L
.32857 .07026 0 -0.96154 Mabsadd L
.32857 .07026 1.46946 -2.02254 Mabsadd L
.32857 .07026 .91448 -0.29713 Mabsadd L
.32857 .07026 2.37764 .77254 Mabsadd L
.32857 .07026 .56518 .7779 Mabsadd L
.32857 .07026 0 2.5 Mabsadd L closepath F
.32952 .07023 0 2.5 Mabsadd m
.32952 .07023 -0.56518 .7779 Mabsadd L
.32952 .07023 -2.37764 .77254 Mabsadd L
.32952 .07023 -0.91448 -0.29713 Mabsadd L
.32952 .07023 -1.46946 -2.02254 Mabsadd L
.32952 .07023 0 -0.96154 Mabsadd L
.32952 .07023 1.46946 -2.02254 Mabsadd L
.32952 .07023 .91448 -0.29713 Mabsadd L
.32952 .07023 2.37764 .77254 Mabsadd L
.32952 .07023 .56518 .7779 Mabsadd L
.32952 .07023 0 2.5 Mabsadd L closepath F
.33048 .07014 0 2.5 Mabsadd m
.33048 .07014 -0.56518 .7779 Mabsadd L
.33048 .07014 -2.37764 .77254 Mabsadd L
.33048 .07014 -0.91448 -0.29713 Mabsadd L
.33048 .07014 -1.46946 -2.02254 Mabsadd L
.33048 .07014 0 -0.96154 Mabsadd L
.33048 .07014 1.46946 -2.02254 Mabsadd L
.33048 .07014 .91448 -0.29713 Mabsadd L
.33048 .07014 2.37764 .77254 Mabsadd L
.33048 .07014 .56518 .7779 Mabsadd L
.33048 .07014 0 2.5 Mabsadd L closepath F
.33143 .07013 0 2.5 Mabsadd m
.33143 .07013 -0.56518 .7779 Mabsadd L
.33143 .07013 -2.37764 .77254 Mabsadd L
.33143 .07013 -0.91448 -0.29713 Mabsadd L
.33143 .07013 -1.46946 -2.02254 Mabsadd L
.33143 .07013 0 -0.96154 Mabsadd L
.33143 .07013 1.46946 -2.02254 Mabsadd L
.33143 .07013 .91448 -0.29713 Mabsadd L
.33143 .07013 2.37764 .77254 Mabsadd L
.33143 .07013 .56518 .7779 Mabsadd L
.33143 .07013 0 2.5 Mabsadd L closepath F
.33238 .06997 0 2.5 Mabsadd m
.33238 .06997 -0.56518 .7779 Mabsadd L
.33238 .06997 -2.37764 .77254 Mabsadd L
.33238 .06997 -0.91448 -0.29713 Mabsadd L
.33238 .06997 -1.46946 -2.02254 Mabsadd L
.33238 .06997 0 -0.96154 Mabsadd L
.33238 .06997 1.46946 -2.02254 Mabsadd L
.33238 .06997 .91448 -0.29713 Mabsadd L
.33238 .06997 2.37764 .77254 Mabsadd L
.33238 .06997 .56518 .7779 Mabsadd L
.33238 .06997 0 2.5 Mabsadd L closepath F
.33333 .06989 0 2.5 Mabsadd m
.33333 .06989 -0.56518 .7779 Mabsadd L
.33333 .06989 -2.37764 .77254 Mabsadd L
.33333 .06989 -0.91448 -0.29713 Mabsadd L
.33333 .06989 -1.46946 -2.02254 Mabsadd L
.33333 .06989 0 -0.96154 Mabsadd L
.33333 .06989 1.46946 -2.02254 Mabsadd L
.33333 .06989 .91448 -0.29713 Mabsadd L
.33333 .06989 2.37764 .77254 Mabsadd L
.33333 .06989 .56518 .7779 Mabsadd L
.33333 .06989 0 2.5 Mabsadd L closepath F
.33429 .06915 0 2.5 Mabsadd m
.33429 .06915 -0.56518 .7779 Mabsadd L
.33429 .06915 -2.37764 .77254 Mabsadd L
.33429 .06915 -0.91448 -0.29713 Mabsadd L
.33429 .06915 -1.46946 -2.02254 Mabsadd L
.33429 .06915 0 -0.96154 Mabsadd L
.33429 .06915 1.46946 -2.02254 Mabsadd L
.33429 .06915 .91448 -0.29713 Mabsadd L
.33429 .06915 2.37764 .77254 Mabsadd L
.33429 .06915 .56518 .7779 Mabsadd L
.33429 .06915 0 2.5 Mabsadd L closepath F
.33524 .06872 0 2.5 Mabsadd m
.33524 .06872 -0.56518 .7779 Mabsadd L
.33524 .06872 -2.37764 .77254 Mabsadd L
.33524 .06872 -0.91448 -0.29713 Mabsadd L
.33524 .06872 -1.46946 -2.02254 Mabsadd L
.33524 .06872 0 -0.96154 Mabsadd L
.33524 .06872 1.46946 -2.02254 Mabsadd L
.33524 .06872 .91448 -0.29713 Mabsadd L
.33524 .06872 2.37764 .77254 Mabsadd L
.33524 .06872 .56518 .7779 Mabsadd L
.33524 .06872 0 2.5 Mabsadd L closepath F
.33619 .06407 0 2.5 Mabsadd m
.33619 .06407 -0.56518 .7779 Mabsadd L
.33619 .06407 -2.37764 .77254 Mabsadd L
.33619 .06407 -0.91448 -0.29713 Mabsadd L
.33619 .06407 -1.46946 -2.02254 Mabsadd L
.33619 .06407 0 -0.96154 Mabsadd L
.33619 .06407 1.46946 -2.02254 Mabsadd L
.33619 .06407 .91448 -0.29713 Mabsadd L
.33619 .06407 2.37764 .77254 Mabsadd L
.33619 .06407 .56518 .7779 Mabsadd L
.33619 .06407 0 2.5 Mabsadd L closepath F
.33714 .06073 0 2.5 Mabsadd m
.33714 .06073 -0.56518 .7779 Mabsadd L
.33714 .06073 -2.37764 .77254 Mabsadd L
.33714 .06073 -0.91448 -0.29713 Mabsadd L
.33714 .06073 -1.46946 -2.02254 Mabsadd L
.33714 .06073 0 -0.96154 Mabsadd L
.33714 .06073 1.46946 -2.02254 Mabsadd L
.33714 .06073 .91448 -0.29713 Mabsadd L
.33714 .06073 2.37764 .77254 Mabsadd L
.33714 .06073 .56518 .7779 Mabsadd L
.33714 .06073 0 2.5 Mabsadd L closepath F
.3381 .0499 0 2.5 Mabsadd m
.3381 .0499 -0.56518 .7779 Mabsadd L
.3381 .0499 -2.37764 .77254 Mabsadd L
.3381 .0499 -0.91448 -0.29713 Mabsadd L
.3381 .0499 -1.46946 -2.02254 Mabsadd L
.3381 .0499 0 -0.96154 Mabsadd L
.3381 .0499 1.46946 -2.02254 Mabsadd L
.3381 .0499 .91448 -0.29713 Mabsadd L
.3381 .0499 2.37764 .77254 Mabsadd L
.3381 .0499 .56518 .7779 Mabsadd L
.3381 .0499 0 2.5 Mabsadd L closepath F
.33905 .03666 0 2.5 Mabsadd m
.33905 .03666 -0.56518 .7779 Mabsadd L
.33905 .03666 -2.37764 .77254 Mabsadd L
.33905 .03666 -0.91448 -0.29713 Mabsadd L
.33905 .03666 -1.46946 -2.02254 Mabsadd L
.33905 .03666 0 -0.96154 Mabsadd L
.33905 .03666 1.46946 -2.02254 Mabsadd L
.33905 .03666 .91448 -0.29713 Mabsadd L
.33905 .03666 2.37764 .77254 Mabsadd L
.33905 .03666 .56518 .7779 Mabsadd L
.33905 .03666 0 2.5 Mabsadd L closepath F
.34 .03664 0 2.5 Mabsadd m
.34 .03664 -0.56518 .7779 Mabsadd L
.34 .03664 -2.37764 .77254 Mabsadd L
.34 .03664 -0.91448 -0.29713 Mabsadd L
.34 .03664 -1.46946 -2.02254 Mabsadd L
.34 .03664 0 -0.96154 Mabsadd L
.34 .03664 1.46946 -2.02254 Mabsadd L
.34 .03664 .91448 -0.29713 Mabsadd L
.34 .03664 2.37764 .77254 Mabsadd L
.34 .03664 .56518 .7779 Mabsadd L
.34 .03664 0 2.5 Mabsadd L closepath F
.34095 .03495 0 2.5 Mabsadd m
.34095 .03495 -0.56518 .7779 Mabsadd L
.34095 .03495 -2.37764 .77254 Mabsadd L
.34095 .03495 -0.91448 -0.29713 Mabsadd L
.34095 .03495 -1.46946 -2.02254 Mabsadd L
.34095 .03495 0 -0.96154 Mabsadd L
.34095 .03495 1.46946 -2.02254 Mabsadd L
.34095 .03495 .91448 -0.29713 Mabsadd L
.34095 .03495 2.37764 .77254 Mabsadd L
.34095 .03495 .56518 .7779 Mabsadd L
.34095 .03495 0 2.5 Mabsadd L closepath F
.3419 .03186 0 2.5 Mabsadd m
.3419 .03186 -0.56518 .7779 Mabsadd L
.3419 .03186 -2.37764 .77254 Mabsadd L
.3419 .03186 -0.91448 -0.29713 Mabsadd L
.3419 .03186 -1.46946 -2.02254 Mabsadd L
.3419 .03186 0 -0.96154 Mabsadd L
.3419 .03186 1.46946 -2.02254 Mabsadd L
.3419 .03186 .91448 -0.29713 Mabsadd L
.3419 .03186 2.37764 .77254 Mabsadd L
.3419 .03186 .56518 .7779 Mabsadd L
.3419 .03186 0 2.5 Mabsadd L closepath F
.34286 .03175 0 2.5 Mabsadd m
.34286 .03175 -0.56518 .7779 Mabsadd L
.34286 .03175 -2.37764 .77254 Mabsadd L
.34286 .03175 -0.91448 -0.29713 Mabsadd L
.34286 .03175 -1.46946 -2.02254 Mabsadd L
.34286 .03175 0 -0.96154 Mabsadd L
.34286 .03175 1.46946 -2.02254 Mabsadd L
.34286 .03175 .91448 -0.29713 Mabsadd L
.34286 .03175 2.37764 .77254 Mabsadd L
.34286 .03175 .56518 .7779 Mabsadd L
.34286 .03175 0 2.5 Mabsadd L closepath F
.34381 .03145 0 2.5 Mabsadd m
.34381 .03145 -0.56518 .7779 Mabsadd L
.34381 .03145 -2.37764 .77254 Mabsadd L
.34381 .03145 -0.91448 -0.29713 Mabsadd L
.34381 .03145 -1.46946 -2.02254 Mabsadd L
.34381 .03145 0 -0.96154 Mabsadd L
.34381 .03145 1.46946 -2.02254 Mabsadd L
.34381 .03145 .91448 -0.29713 Mabsadd L
.34381 .03145 2.37764 .77254 Mabsadd L
.34381 .03145 .56518 .7779 Mabsadd L
.34381 .03145 0 2.5 Mabsadd L closepath F
.34476 .02876 0 2.5 Mabsadd m
.34476 .02876 -0.56518 .7779 Mabsadd L
.34476 .02876 -2.37764 .77254 Mabsadd L
.34476 .02876 -0.91448 -0.29713 Mabsadd L
.34476 .02876 -1.46946 -2.02254 Mabsadd L
.34476 .02876 0 -0.96154 Mabsadd L
.34476 .02876 1.46946 -2.02254 Mabsadd L
.34476 .02876 .91448 -0.29713 Mabsadd L
.34476 .02876 2.37764 .77254 Mabsadd L
.34476 .02876 .56518 .7779 Mabsadd L
.34476 .02876 0 2.5 Mabsadd L closepath F
.34571 .02552 0 2.5 Mabsadd m
.34571 .02552 -0.56518 .7779 Mabsadd L
.34571 .02552 -2.37764 .77254 Mabsadd L
.34571 .02552 -0.91448 -0.29713 Mabsadd L
.34571 .02552 -1.46946 -2.02254 Mabsadd L
.34571 .02552 0 -0.96154 Mabsadd L
.34571 .02552 1.46946 -2.02254 Mabsadd L
.34571 .02552 .91448 -0.29713 Mabsadd L
.34571 .02552 2.37764 .77254 Mabsadd L
.34571 .02552 .56518 .7779 Mabsadd L
.34571 .02552 0 2.5 Mabsadd L closepath F
.34667 .02568 0 2.5 Mabsadd m
.34667 .02568 -0.56518 .7779 Mabsadd L
.34667 .02568 -2.37764 .77254 Mabsadd L
.34667 .02568 -0.91448 -0.29713 Mabsadd L
.34667 .02568 -1.46946 -2.02254 Mabsadd L
.34667 .02568 0 -0.96154 Mabsadd L
.34667 .02568 1.46946 -2.02254 Mabsadd L
.34667 .02568 .91448 -0.29713 Mabsadd L
.34667 .02568 2.37764 .77254 Mabsadd L
.34667 .02568 .56518 .7779 Mabsadd L
.34667 .02568 0 2.5 Mabsadd L closepath F
.34762 .0872 0 2.5 Mabsadd m
.34762 .0872 -0.56518 .7779 Mabsadd L
.34762 .0872 -2.37764 .77254 Mabsadd L
.34762 .0872 -0.91448 -0.29713 Mabsadd L
.34762 .0872 -1.46946 -2.02254 Mabsadd L
.34762 .0872 0 -0.96154 Mabsadd L
.34762 .0872 1.46946 -2.02254 Mabsadd L
.34762 .0872 .91448 -0.29713 Mabsadd L
.34762 .0872 2.37764 .77254 Mabsadd L
.34762 .0872 .56518 .7779 Mabsadd L
.34762 .0872 0 2.5 Mabsadd L closepath F
.34857 .09926 0 2.5 Mabsadd m
.34857 .09926 -0.56518 .7779 Mabsadd L
.34857 .09926 -2.37764 .77254 Mabsadd L
.34857 .09926 -0.91448 -0.29713 Mabsadd L
.34857 .09926 -1.46946 -2.02254 Mabsadd L
.34857 .09926 0 -0.96154 Mabsadd L
.34857 .09926 1.46946 -2.02254 Mabsadd L
.34857 .09926 .91448 -0.29713 Mabsadd L
.34857 .09926 2.37764 .77254 Mabsadd L
.34857 .09926 .56518 .7779 Mabsadd L
.34857 .09926 0 2.5 Mabsadd L closepath F
.34952 .10032 0 2.5 Mabsadd m
.34952 .10032 -0.56518 .7779 Mabsadd L
.34952 .10032 -2.37764 .77254 Mabsadd L
.34952 .10032 -0.91448 -0.29713 Mabsadd L
.34952 .10032 -1.46946 -2.02254 Mabsadd L
.34952 .10032 0 -0.96154 Mabsadd L
.34952 .10032 1.46946 -2.02254 Mabsadd L
.34952 .10032 .91448 -0.29713 Mabsadd L
.34952 .10032 2.37764 .77254 Mabsadd L
.34952 .10032 .56518 .7779 Mabsadd L
.34952 .10032 0 2.5 Mabsadd L closepath F
.35048 .10338 0 2.5 Mabsadd m
.35048 .10338 -0.56518 .7779 Mabsadd L
.35048 .10338 -2.37764 .77254 Mabsadd L
.35048 .10338 -0.91448 -0.29713 Mabsadd L
.35048 .10338 -1.46946 -2.02254 Mabsadd L
.35048 .10338 0 -0.96154 Mabsadd L
.35048 .10338 1.46946 -2.02254 Mabsadd L
.35048 .10338 .91448 -0.29713 Mabsadd L
.35048 .10338 2.37764 .77254 Mabsadd L
.35048 .10338 .56518 .7779 Mabsadd L
.35048 .10338 0 2.5 Mabsadd L closepath F
.35143 .10489 0 2.5 Mabsadd m
.35143 .10489 -0.56518 .7779 Mabsadd L
.35143 .10489 -2.37764 .77254 Mabsadd L
.35143 .10489 -0.91448 -0.29713 Mabsadd L
.35143 .10489 -1.46946 -2.02254 Mabsadd L
.35143 .10489 0 -0.96154 Mabsadd L
.35143 .10489 1.46946 -2.02254 Mabsadd L
.35143 .10489 .91448 -0.29713 Mabsadd L
.35143 .10489 2.37764 .77254 Mabsadd L
.35143 .10489 .56518 .7779 Mabsadd L
.35143 .10489 0 2.5 Mabsadd L closepath F
.35238 .10456 0 2.5 Mabsadd m
.35238 .10456 -0.56518 .7779 Mabsadd L
.35238 .10456 -2.37764 .77254 Mabsadd L
.35238 .10456 -0.91448 -0.29713 Mabsadd L
.35238 .10456 -1.46946 -2.02254 Mabsadd L
.35238 .10456 0 -0.96154 Mabsadd L
.35238 .10456 1.46946 -2.02254 Mabsadd L
.35238 .10456 .91448 -0.29713 Mabsadd L
.35238 .10456 2.37764 .77254 Mabsadd L
.35238 .10456 .56518 .7779 Mabsadd L
.35238 .10456 0 2.5 Mabsadd L closepath F
.35333 .11018 0 2.5 Mabsadd m
.35333 .11018 -0.56518 .7779 Mabsadd L
.35333 .11018 -2.37764 .77254 Mabsadd L
.35333 .11018 -0.91448 -0.29713 Mabsadd L
.35333 .11018 -1.46946 -2.02254 Mabsadd L
.35333 .11018 0 -0.96154 Mabsadd L
.35333 .11018 1.46946 -2.02254 Mabsadd L
.35333 .11018 .91448 -0.29713 Mabsadd L
.35333 .11018 2.37764 .77254 Mabsadd L
.35333 .11018 .56518 .7779 Mabsadd L
.35333 .11018 0 2.5 Mabsadd L closepath F
.35429 .11215 0 2.5 Mabsadd m
.35429 .11215 -0.56518 .7779 Mabsadd L
.35429 .11215 -2.37764 .77254 Mabsadd L
.35429 .11215 -0.91448 -0.29713 Mabsadd L
.35429 .11215 -1.46946 -2.02254 Mabsadd L
.35429 .11215 0 -0.96154 Mabsadd L
.35429 .11215 1.46946 -2.02254 Mabsadd L
.35429 .11215 .91448 -0.29713 Mabsadd L
.35429 .11215 2.37764 .77254 Mabsadd L
.35429 .11215 .56518 .7779 Mabsadd L
.35429 .11215 0 2.5 Mabsadd L closepath F
.35524 .11793 0 2.5 Mabsadd m
.35524 .11793 -0.56518 .7779 Mabsadd L
.35524 .11793 -2.37764 .77254 Mabsadd L
.35524 .11793 -0.91448 -0.29713 Mabsadd L
.35524 .11793 -1.46946 -2.02254 Mabsadd L
.35524 .11793 0 -0.96154 Mabsadd L
.35524 .11793 1.46946 -2.02254 Mabsadd L
.35524 .11793 .91448 -0.29713 Mabsadd L
.35524 .11793 2.37764 .77254 Mabsadd L
.35524 .11793 .56518 .7779 Mabsadd L
.35524 .11793 0 2.5 Mabsadd L closepath F
.35619 .11821 0 2.5 Mabsadd m
.35619 .11821 -0.56518 .7779 Mabsadd L
.35619 .11821 -2.37764 .77254 Mabsadd L
.35619 .11821 -0.91448 -0.29713 Mabsadd L
.35619 .11821 -1.46946 -2.02254 Mabsadd L
.35619 .11821 0 -0.96154 Mabsadd L
.35619 .11821 1.46946 -2.02254 Mabsadd L
.35619 .11821 .91448 -0.29713 Mabsadd L
.35619 .11821 2.37764 .77254 Mabsadd L
.35619 .11821 .56518 .7779 Mabsadd L
.35619 .11821 0 2.5 Mabsadd L closepath F
.35714 .11912 0 2.5 Mabsadd m
.35714 .11912 -0.56518 .7779 Mabsadd L
.35714 .11912 -2.37764 .77254 Mabsadd L
.35714 .11912 -0.91448 -0.29713 Mabsadd L
.35714 .11912 -1.46946 -2.02254 Mabsadd L
.35714 .11912 0 -0.96154 Mabsadd L
.35714 .11912 1.46946 -2.02254 Mabsadd L
.35714 .11912 .91448 -0.29713 Mabsadd L
.35714 .11912 2.37764 .77254 Mabsadd L
.35714 .11912 .56518 .7779 Mabsadd L
.35714 .11912 0 2.5 Mabsadd L closepath F
.3581 .11918 0 2.5 Mabsadd m
.3581 .11918 -0.56518 .7779 Mabsadd L
.3581 .11918 -2.37764 .77254 Mabsadd L
.3581 .11918 -0.91448 -0.29713 Mabsadd L
.3581 .11918 -1.46946 -2.02254 Mabsadd L
.3581 .11918 0 -0.96154 Mabsadd L
.3581 .11918 1.46946 -2.02254 Mabsadd L
.3581 .11918 .91448 -0.29713 Mabsadd L
.3581 .11918 2.37764 .77254 Mabsadd L
.3581 .11918 .56518 .7779 Mabsadd L
.3581 .11918 0 2.5 Mabsadd L closepath F
.35905 .11937 0 2.5 Mabsadd m
.35905 .11937 -0.56518 .7779 Mabsadd L
.35905 .11937 -2.37764 .77254 Mabsadd L
.35905 .11937 -0.91448 -0.29713 Mabsadd L
.35905 .11937 -1.46946 -2.02254 Mabsadd L
.35905 .11937 0 -0.96154 Mabsadd L
.35905 .11937 1.46946 -2.02254 Mabsadd L
.35905 .11937 .91448 -0.29713 Mabsadd L
.35905 .11937 2.37764 .77254 Mabsadd L
.35905 .11937 .56518 .7779 Mabsadd L
.35905 .11937 0 2.5 Mabsadd L closepath F
.36 .11966 0 2.5 Mabsadd m
.36 .11966 -0.56518 .7779 Mabsadd L
.36 .11966 -2.37764 .77254 Mabsadd L
.36 .11966 -0.91448 -0.29713 Mabsadd L
.36 .11966 -1.46946 -2.02254 Mabsadd L
.36 .11966 0 -0.96154 Mabsadd L
.36 .11966 1.46946 -2.02254 Mabsadd L
.36 .11966 .91448 -0.29713 Mabsadd L
.36 .11966 2.37764 .77254 Mabsadd L
.36 .11966 .56518 .7779 Mabsadd L
.36 .11966 0 2.5 Mabsadd L closepath F
.36095 .11967 0 2.5 Mabsadd m
.36095 .11967 -0.56518 .7779 Mabsadd L
.36095 .11967 -2.37764 .77254 Mabsadd L
.36095 .11967 -0.91448 -0.29713 Mabsadd L
.36095 .11967 -1.46946 -2.02254 Mabsadd L
.36095 .11967 0 -0.96154 Mabsadd L
.36095 .11967 1.46946 -2.02254 Mabsadd L
.36095 .11967 .91448 -0.29713 Mabsadd L
.36095 .11967 2.37764 .77254 Mabsadd L
.36095 .11967 .56518 .7779 Mabsadd L
.36095 .11967 0 2.5 Mabsadd L closepath F
.3619 .11969 0 2.5 Mabsadd m
.3619 .11969 -0.56518 .7779 Mabsadd L
.3619 .11969 -2.37764 .77254 Mabsadd L
.3619 .11969 -0.91448 -0.29713 Mabsadd L
.3619 .11969 -1.46946 -2.02254 Mabsadd L
.3619 .11969 0 -0.96154 Mabsadd L
.3619 .11969 1.46946 -2.02254 Mabsadd L
.3619 .11969 .91448 -0.29713 Mabsadd L
.3619 .11969 2.37764 .77254 Mabsadd L
.3619 .11969 .56518 .7779 Mabsadd L
.3619 .11969 0 2.5 Mabsadd L closepath F
.36286 .11974 0 2.5 Mabsadd m
.36286 .11974 -0.56518 .7779 Mabsadd L
.36286 .11974 -2.37764 .77254 Mabsadd L
.36286 .11974 -0.91448 -0.29713 Mabsadd L
.36286 .11974 -1.46946 -2.02254 Mabsadd L
.36286 .11974 0 -0.96154 Mabsadd L
.36286 .11974 1.46946 -2.02254 Mabsadd L
.36286 .11974 .91448 -0.29713 Mabsadd L
.36286 .11974 2.37764 .77254 Mabsadd L
.36286 .11974 .56518 .7779 Mabsadd L
.36286 .11974 0 2.5 Mabsadd L closepath F
.36381 .11974 0 2.5 Mabsadd m
.36381 .11974 -0.56518 .7779 Mabsadd L
.36381 .11974 -2.37764 .77254 Mabsadd L
.36381 .11974 -0.91448 -0.29713 Mabsadd L
.36381 .11974 -1.46946 -2.02254 Mabsadd L
.36381 .11974 0 -0.96154 Mabsadd L
.36381 .11974 1.46946 -2.02254 Mabsadd L
.36381 .11974 .91448 -0.29713 Mabsadd L
.36381 .11974 2.37764 .77254 Mabsadd L
.36381 .11974 .56518 .7779 Mabsadd L
.36381 .11974 0 2.5 Mabsadd L closepath F
.36476 .11975 0 2.5 Mabsadd m
.36476 .11975 -0.56518 .7779 Mabsadd L
.36476 .11975 -2.37764 .77254 Mabsadd L
.36476 .11975 -0.91448 -0.29713 Mabsadd L
.36476 .11975 -1.46946 -2.02254 Mabsadd L
.36476 .11975 0 -0.96154 Mabsadd L
.36476 .11975 1.46946 -2.02254 Mabsadd L
.36476 .11975 .91448 -0.29713 Mabsadd L
.36476 .11975 2.37764 .77254 Mabsadd L
.36476 .11975 .56518 .7779 Mabsadd L
.36476 .11975 0 2.5 Mabsadd L closepath F
.36571 .11976 0 2.5 Mabsadd m
.36571 .11976 -0.56518 .7779 Mabsadd L
.36571 .11976 -2.37764 .77254 Mabsadd L
.36571 .11976 -0.91448 -0.29713 Mabsadd L
.36571 .11976 -1.46946 -2.02254 Mabsadd L
.36571 .11976 0 -0.96154 Mabsadd L
.36571 .11976 1.46946 -2.02254 Mabsadd L
.36571 .11976 .91448 -0.29713 Mabsadd L
.36571 .11976 2.37764 .77254 Mabsadd L
.36571 .11976 .56518 .7779 Mabsadd L
.36571 .11976 0 2.5 Mabsadd L closepath F
.36667 .11976 0 2.5 Mabsadd m
.36667 .11976 -0.56518 .7779 Mabsadd L
.36667 .11976 -2.37764 .77254 Mabsadd L
.36667 .11976 -0.91448 -0.29713 Mabsadd L
.36667 .11976 -1.46946 -2.02254 Mabsadd L
.36667 .11976 0 -0.96154 Mabsadd L
.36667 .11976 1.46946 -2.02254 Mabsadd L
.36667 .11976 .91448 -0.29713 Mabsadd L
.36667 .11976 2.37764 .77254 Mabsadd L
.36667 .11976 .56518 .7779 Mabsadd L
.36667 .11976 0 2.5 Mabsadd L closepath F
.36762 .11976 0 2.5 Mabsadd m
.36762 .11976 -0.56518 .7779 Mabsadd L
.36762 .11976 -2.37764 .77254 Mabsadd L
.36762 .11976 -0.91448 -0.29713 Mabsadd L
.36762 .11976 -1.46946 -2.02254 Mabsadd L
.36762 .11976 0 -0.96154 Mabsadd L
.36762 .11976 1.46946 -2.02254 Mabsadd L
.36762 .11976 .91448 -0.29713 Mabsadd L
.36762 .11976 2.37764 .77254 Mabsadd L
.36762 .11976 .56518 .7779 Mabsadd L
.36762 .11976 0 2.5 Mabsadd L closepath F
.36857 .11976 0 2.5 Mabsadd m
.36857 .11976 -0.56518 .7779 Mabsadd L
.36857 .11976 -2.37764 .77254 Mabsadd L
.36857 .11976 -0.91448 -0.29713 Mabsadd L
.36857 .11976 -1.46946 -2.02254 Mabsadd L
.36857 .11976 0 -0.96154 Mabsadd L
.36857 .11976 1.46946 -2.02254 Mabsadd L
.36857 .11976 .91448 -0.29713 Mabsadd L
.36857 .11976 2.37764 .77254 Mabsadd L
.36857 .11976 .56518 .7779 Mabsadd L
.36857 .11976 0 2.5 Mabsadd L closepath F
.36952 .11976 0 2.5 Mabsadd m
.36952 .11976 -0.56518 .7779 Mabsadd L
.36952 .11976 -2.37764 .77254 Mabsadd L
.36952 .11976 -0.91448 -0.29713 Mabsadd L
.36952 .11976 -1.46946 -2.02254 Mabsadd L
.36952 .11976 0 -0.96154 Mabsadd L
.36952 .11976 1.46946 -2.02254 Mabsadd L
.36952 .11976 .91448 -0.29713 Mabsadd L
.36952 .11976 2.37764 .77254 Mabsadd L
.36952 .11976 .56518 .7779 Mabsadd L
.36952 .11976 0 2.5 Mabsadd L closepath F
.37048 .11976 0 2.5 Mabsadd m
.37048 .11976 -0.56518 .7779 Mabsadd L
.37048 .11976 -2.37764 .77254 Mabsadd L
.37048 .11976 -0.91448 -0.29713 Mabsadd L
.37048 .11976 -1.46946 -2.02254 Mabsadd L
.37048 .11976 0 -0.96154 Mabsadd L
.37048 .11976 1.46946 -2.02254 Mabsadd L
.37048 .11976 .91448 -0.29713 Mabsadd L
.37048 .11976 2.37764 .77254 Mabsadd L
.37048 .11976 .56518 .7779 Mabsadd L
.37048 .11976 0 2.5 Mabsadd L closepath F
.37143 .11976 0 2.5 Mabsadd m
.37143 .11976 -0.56518 .7779 Mabsadd L
.37143 .11976 -2.37764 .77254 Mabsadd L
.37143 .11976 -0.91448 -0.29713 Mabsadd L
.37143 .11976 -1.46946 -2.02254 Mabsadd L
.37143 .11976 0 -0.96154 Mabsadd L
.37143 .11976 1.46946 -2.02254 Mabsadd L
.37143 .11976 .91448 -0.29713 Mabsadd L
.37143 .11976 2.37764 .77254 Mabsadd L
.37143 .11976 .56518 .7779 Mabsadd L
.37143 .11976 0 2.5 Mabsadd L closepath F
.37238 .11976 0 2.5 Mabsadd m
.37238 .11976 -0.56518 .7779 Mabsadd L
.37238 .11976 -2.37764 .77254 Mabsadd L
.37238 .11976 -0.91448 -0.29713 Mabsadd L
.37238 .11976 -1.46946 -2.02254 Mabsadd L
.37238 .11976 0 -0.96154 Mabsadd L
.37238 .11976 1.46946 -2.02254 Mabsadd L
.37238 .11976 .91448 -0.29713 Mabsadd L
.37238 .11976 2.37764 .77254 Mabsadd L
.37238 .11976 .56518 .7779 Mabsadd L
.37238 .11976 0 2.5 Mabsadd L closepath F
.37333 .11976 0 2.5 Mabsadd m
.37333 .11976 -0.56518 .7779 Mabsadd L
.37333 .11976 -2.37764 .77254 Mabsadd L
.37333 .11976 -0.91448 -0.29713 Mabsadd L
.37333 .11976 -1.46946 -2.02254 Mabsadd L
.37333 .11976 0 -0.96154 Mabsadd L
.37333 .11976 1.46946 -2.02254 Mabsadd L
.37333 .11976 .91448 -0.29713 Mabsadd L
.37333 .11976 2.37764 .77254 Mabsadd L
.37333 .11976 .56518 .7779 Mabsadd L
.37333 .11976 0 2.5 Mabsadd L closepath F
.37429 .11976 0 2.5 Mabsadd m
.37429 .11976 -0.56518 .7779 Mabsadd L
.37429 .11976 -2.37764 .77254 Mabsadd L
.37429 .11976 -0.91448 -0.29713 Mabsadd L
.37429 .11976 -1.46946 -2.02254 Mabsadd L
.37429 .11976 0 -0.96154 Mabsadd L
.37429 .11976 1.46946 -2.02254 Mabsadd L
.37429 .11976 .91448 -0.29713 Mabsadd L
.37429 .11976 2.37764 .77254 Mabsadd L
.37429 .11976 .56518 .7779 Mabsadd L
.37429 .11976 0 2.5 Mabsadd L closepath F
.37524 .11976 0 2.5 Mabsadd m
.37524 .11976 -0.56518 .7779 Mabsadd L
.37524 .11976 -2.37764 .77254 Mabsadd L
.37524 .11976 -0.91448 -0.29713 Mabsadd L
.37524 .11976 -1.46946 -2.02254 Mabsadd L
.37524 .11976 0 -0.96154 Mabsadd L
.37524 .11976 1.46946 -2.02254 Mabsadd L
.37524 .11976 .91448 -0.29713 Mabsadd L
.37524 .11976 2.37764 .77254 Mabsadd L
.37524 .11976 .56518 .7779 Mabsadd L
.37524 .11976 0 2.5 Mabsadd L closepath F
.37619 .11976 0 2.5 Mabsadd m
.37619 .11976 -0.56518 .7779 Mabsadd L
.37619 .11976 -2.37764 .77254 Mabsadd L
.37619 .11976 -0.91448 -0.29713 Mabsadd L
.37619 .11976 -1.46946 -2.02254 Mabsadd L
.37619 .11976 0 -0.96154 Mabsadd L
.37619 .11976 1.46946 -2.02254 Mabsadd L
.37619 .11976 .91448 -0.29713 Mabsadd L
.37619 .11976 2.37764 .77254 Mabsadd L
.37619 .11976 .56518 .7779 Mabsadd L
.37619 .11976 0 2.5 Mabsadd L closepath F
.37714 .11975 0 2.5 Mabsadd m
.37714 .11975 -0.56518 .7779 Mabsadd L
.37714 .11975 -2.37764 .77254 Mabsadd L
.37714 .11975 -0.91448 -0.29713 Mabsadd L
.37714 .11975 -1.46946 -2.02254 Mabsadd L
.37714 .11975 0 -0.96154 Mabsadd L
.37714 .11975 1.46946 -2.02254 Mabsadd L
.37714 .11975 .91448 -0.29713 Mabsadd L
.37714 .11975 2.37764 .77254 Mabsadd L
.37714 .11975 .56518 .7779 Mabsadd L
.37714 .11975 0 2.5 Mabsadd L closepath F
.3781 .11975 0 2.5 Mabsadd m
.3781 .11975 -0.56518 .7779 Mabsadd L
.3781 .11975 -2.37764 .77254 Mabsadd L
.3781 .11975 -0.91448 -0.29713 Mabsadd L
.3781 .11975 -1.46946 -2.02254 Mabsadd L
.3781 .11975 0 -0.96154 Mabsadd L
.3781 .11975 1.46946 -2.02254 Mabsadd L
.3781 .11975 .91448 -0.29713 Mabsadd L
.3781 .11975 2.37764 .77254 Mabsadd L
.3781 .11975 .56518 .7779 Mabsadd L
.3781 .11975 0 2.5 Mabsadd L closepath F
.37905 .11974 0 2.5 Mabsadd m
.37905 .11974 -0.56518 .7779 Mabsadd L
.37905 .11974 -2.37764 .77254 Mabsadd L
.37905 .11974 -0.91448 -0.29713 Mabsadd L
.37905 .11974 -1.46946 -2.02254 Mabsadd L
.37905 .11974 0 -0.96154 Mabsadd L
.37905 .11974 1.46946 -2.02254 Mabsadd L
.37905 .11974 .91448 -0.29713 Mabsadd L
.37905 .11974 2.37764 .77254 Mabsadd L
.37905 .11974 .56518 .7779 Mabsadd L
.37905 .11974 0 2.5 Mabsadd L closepath F
.38 .11974 0 2.5 Mabsadd m
.38 .11974 -0.56518 .7779 Mabsadd L
.38 .11974 -2.37764 .77254 Mabsadd L
.38 .11974 -0.91448 -0.29713 Mabsadd L
.38 .11974 -1.46946 -2.02254 Mabsadd L
.38 .11974 0 -0.96154 Mabsadd L
.38 .11974 1.46946 -2.02254 Mabsadd L
.38 .11974 .91448 -0.29713 Mabsadd L
.38 .11974 2.37764 .77254 Mabsadd L
.38 .11974 .56518 .7779 Mabsadd L
.38 .11974 0 2.5 Mabsadd L closepath F
.38095 .11973 0 2.5 Mabsadd m
.38095 .11973 -0.56518 .7779 Mabsadd L
.38095 .11973 -2.37764 .77254 Mabsadd L
.38095 .11973 -0.91448 -0.29713 Mabsadd L
.38095 .11973 -1.46946 -2.02254 Mabsadd L
.38095 .11973 0 -0.96154 Mabsadd L
.38095 .11973 1.46946 -2.02254 Mabsadd L
.38095 .11973 .91448 -0.29713 Mabsadd L
.38095 .11973 2.37764 .77254 Mabsadd L
.38095 .11973 .56518 .7779 Mabsadd L
.38095 .11973 0 2.5 Mabsadd L closepath F
.3819 .1197 0 2.5 Mabsadd m
.3819 .1197 -0.56518 .7779 Mabsadd L
.3819 .1197 -2.37764 .77254 Mabsadd L
.3819 .1197 -0.91448 -0.29713 Mabsadd L
.3819 .1197 -1.46946 -2.02254 Mabsadd L
.3819 .1197 0 -0.96154 Mabsadd L
.3819 .1197 1.46946 -2.02254 Mabsadd L
.3819 .1197 .91448 -0.29713 Mabsadd L
.3819 .1197 2.37764 .77254 Mabsadd L
.3819 .1197 .56518 .7779 Mabsadd L
.3819 .1197 0 2.5 Mabsadd L closepath F
.38286 .1197 0 2.5 Mabsadd m
.38286 .1197 -0.56518 .7779 Mabsadd L
.38286 .1197 -2.37764 .77254 Mabsadd L
.38286 .1197 -0.91448 -0.29713 Mabsadd L
.38286 .1197 -1.46946 -2.02254 Mabsadd L
.38286 .1197 0 -0.96154 Mabsadd L
.38286 .1197 1.46946 -2.02254 Mabsadd L
.38286 .1197 .91448 -0.29713 Mabsadd L
.38286 .1197 2.37764 .77254 Mabsadd L
.38286 .1197 .56518 .7779 Mabsadd L
.38286 .1197 0 2.5 Mabsadd L closepath F
.38381 .11966 0 2.5 Mabsadd m
.38381 .11966 -0.56518 .7779 Mabsadd L
.38381 .11966 -2.37764 .77254 Mabsadd L
.38381 .11966 -0.91448 -0.29713 Mabsadd L
.38381 .11966 -1.46946 -2.02254 Mabsadd L
.38381 .11966 0 -0.96154 Mabsadd L
.38381 .11966 1.46946 -2.02254 Mabsadd L
.38381 .11966 .91448 -0.29713 Mabsadd L
.38381 .11966 2.37764 .77254 Mabsadd L
.38381 .11966 .56518 .7779 Mabsadd L
.38381 .11966 0 2.5 Mabsadd L closepath F
.38476 .11951 0 2.5 Mabsadd m
.38476 .11951 -0.56518 .7779 Mabsadd L
.38476 .11951 -2.37764 .77254 Mabsadd L
.38476 .11951 -0.91448 -0.29713 Mabsadd L
.38476 .11951 -1.46946 -2.02254 Mabsadd L
.38476 .11951 0 -0.96154 Mabsadd L
.38476 .11951 1.46946 -2.02254 Mabsadd L
.38476 .11951 .91448 -0.29713 Mabsadd L
.38476 .11951 2.37764 .77254 Mabsadd L
.38476 .11951 .56518 .7779 Mabsadd L
.38476 .11951 0 2.5 Mabsadd L closepath F
.38571 .11948 0 2.5 Mabsadd m
.38571 .11948 -0.56518 .7779 Mabsadd L
.38571 .11948 -2.37764 .77254 Mabsadd L
.38571 .11948 -0.91448 -0.29713 Mabsadd L
.38571 .11948 -1.46946 -2.02254 Mabsadd L
.38571 .11948 0 -0.96154 Mabsadd L
.38571 .11948 1.46946 -2.02254 Mabsadd L
.38571 .11948 .91448 -0.29713 Mabsadd L
.38571 .11948 2.37764 .77254 Mabsadd L
.38571 .11948 .56518 .7779 Mabsadd L
.38571 .11948 0 2.5 Mabsadd L closepath F
.38667 .11917 0 2.5 Mabsadd m
.38667 .11917 -0.56518 .7779 Mabsadd L
.38667 .11917 -2.37764 .77254 Mabsadd L
.38667 .11917 -0.91448 -0.29713 Mabsadd L
.38667 .11917 -1.46946 -2.02254 Mabsadd L
.38667 .11917 0 -0.96154 Mabsadd L
.38667 .11917 1.46946 -2.02254 Mabsadd L
.38667 .11917 .91448 -0.29713 Mabsadd L
.38667 .11917 2.37764 .77254 Mabsadd L
.38667 .11917 .56518 .7779 Mabsadd L
.38667 .11917 0 2.5 Mabsadd L closepath F
.38762 .11902 0 2.5 Mabsadd m
.38762 .11902 -0.56518 .7779 Mabsadd L
.38762 .11902 -2.37764 .77254 Mabsadd L
.38762 .11902 -0.91448 -0.29713 Mabsadd L
.38762 .11902 -1.46946 -2.02254 Mabsadd L
.38762 .11902 0 -0.96154 Mabsadd L
.38762 .11902 1.46946 -2.02254 Mabsadd L
.38762 .11902 .91448 -0.29713 Mabsadd L
.38762 .11902 2.37764 .77254 Mabsadd L
.38762 .11902 .56518 .7779 Mabsadd L
.38762 .11902 0 2.5 Mabsadd L closepath F
.38857 .11743 0 2.5 Mabsadd m
.38857 .11743 -0.56518 .7779 Mabsadd L
.38857 .11743 -2.37764 .77254 Mabsadd L
.38857 .11743 -0.91448 -0.29713 Mabsadd L
.38857 .11743 -1.46946 -2.02254 Mabsadd L
.38857 .11743 0 -0.96154 Mabsadd L
.38857 .11743 1.46946 -2.02254 Mabsadd L
.38857 .11743 .91448 -0.29713 Mabsadd L
.38857 .11743 2.37764 .77254 Mabsadd L
.38857 .11743 .56518 .7779 Mabsadd L
.38857 .11743 0 2.5 Mabsadd L closepath F
.38952 .11643 0 2.5 Mabsadd m
.38952 .11643 -0.56518 .7779 Mabsadd L
.38952 .11643 -2.37764 .77254 Mabsadd L
.38952 .11643 -0.91448 -0.29713 Mabsadd L
.38952 .11643 -1.46946 -2.02254 Mabsadd L
.38952 .11643 0 -0.96154 Mabsadd L
.38952 .11643 1.46946 -2.02254 Mabsadd L
.38952 .11643 .91448 -0.29713 Mabsadd L
.38952 .11643 2.37764 .77254 Mabsadd L
.38952 .11643 .56518 .7779 Mabsadd L
.38952 .11643 0 2.5 Mabsadd L closepath F
.39048 .11354 0 2.5 Mabsadd m
.39048 .11354 -0.56518 .7779 Mabsadd L
.39048 .11354 -2.37764 .77254 Mabsadd L
.39048 .11354 -0.91448 -0.29713 Mabsadd L
.39048 .11354 -1.46946 -2.02254 Mabsadd L
.39048 .11354 0 -0.96154 Mabsadd L
.39048 .11354 1.46946 -2.02254 Mabsadd L
.39048 .11354 .91448 -0.29713 Mabsadd L
.39048 .11354 2.37764 .77254 Mabsadd L
.39048 .11354 .56518 .7779 Mabsadd L
.39048 .11354 0 2.5 Mabsadd L closepath F
.39143 .10429 0 2.5 Mabsadd m
.39143 .10429 -0.56518 .7779 Mabsadd L
.39143 .10429 -2.37764 .77254 Mabsadd L
.39143 .10429 -0.91448 -0.29713 Mabsadd L
.39143 .10429 -1.46946 -2.02254 Mabsadd L
.39143 .10429 0 -0.96154 Mabsadd L
.39143 .10429 1.46946 -2.02254 Mabsadd L
.39143 .10429 .91448 -0.29713 Mabsadd L
.39143 .10429 2.37764 .77254 Mabsadd L
.39143 .10429 .56518 .7779 Mabsadd L
.39143 .10429 0 2.5 Mabsadd L closepath F
.39238 .07148 0 2.5 Mabsadd m
.39238 .07148 -0.56518 .7779 Mabsadd L
.39238 .07148 -2.37764 .77254 Mabsadd L
.39238 .07148 -0.91448 -0.29713 Mabsadd L
.39238 .07148 -1.46946 -2.02254 Mabsadd L
.39238 .07148 0 -0.96154 Mabsadd L
.39238 .07148 1.46946 -2.02254 Mabsadd L
.39238 .07148 .91448 -0.29713 Mabsadd L
.39238 .07148 2.37764 .77254 Mabsadd L
.39238 .07148 .56518 .7779 Mabsadd L
.39238 .07148 0 2.5 Mabsadd L closepath F
.39333 .07149 0 2.5 Mabsadd m
.39333 .07149 -0.56518 .7779 Mabsadd L
.39333 .07149 -2.37764 .77254 Mabsadd L
.39333 .07149 -0.91448 -0.29713 Mabsadd L
.39333 .07149 -1.46946 -2.02254 Mabsadd L
.39333 .07149 0 -0.96154 Mabsadd L
.39333 .07149 1.46946 -2.02254 Mabsadd L
.39333 .07149 .91448 -0.29713 Mabsadd L
.39333 .07149 2.37764 .77254 Mabsadd L
.39333 .07149 .56518 .7779 Mabsadd L
.39333 .07149 0 2.5 Mabsadd L closepath F
.39429 .07274 0 2.5 Mabsadd m
.39429 .07274 -0.56518 .7779 Mabsadd L
.39429 .07274 -2.37764 .77254 Mabsadd L
.39429 .07274 -0.91448 -0.29713 Mabsadd L
.39429 .07274 -1.46946 -2.02254 Mabsadd L
.39429 .07274 0 -0.96154 Mabsadd L
.39429 .07274 1.46946 -2.02254 Mabsadd L
.39429 .07274 .91448 -0.29713 Mabsadd L
.39429 .07274 2.37764 .77254 Mabsadd L
.39429 .07274 .56518 .7779 Mabsadd L
.39429 .07274 0 2.5 Mabsadd L closepath F
.39524 .07493 0 2.5 Mabsadd m
.39524 .07493 -0.56518 .7779 Mabsadd L
.39524 .07493 -2.37764 .77254 Mabsadd L
.39524 .07493 -0.91448 -0.29713 Mabsadd L
.39524 .07493 -1.46946 -2.02254 Mabsadd L
.39524 .07493 0 -0.96154 Mabsadd L
.39524 .07493 1.46946 -2.02254 Mabsadd L
.39524 .07493 .91448 -0.29713 Mabsadd L
.39524 .07493 2.37764 .77254 Mabsadd L
.39524 .07493 .56518 .7779 Mabsadd L
.39524 .07493 0 2.5 Mabsadd L closepath F
.39619 .075 0 2.5 Mabsadd m
.39619 .075 -0.56518 .7779 Mabsadd L
.39619 .075 -2.37764 .77254 Mabsadd L
.39619 .075 -0.91448 -0.29713 Mabsadd L
.39619 .075 -1.46946 -2.02254 Mabsadd L
.39619 .075 0 -0.96154 Mabsadd L
.39619 .075 1.46946 -2.02254 Mabsadd L
.39619 .075 .91448 -0.29713 Mabsadd L
.39619 .075 2.37764 .77254 Mabsadd L
.39619 .075 .56518 .7779 Mabsadd L
.39619 .075 0 2.5 Mabsadd L closepath F
.39714 .07521 0 2.5 Mabsadd m
.39714 .07521 -0.56518 .7779 Mabsadd L
.39714 .07521 -2.37764 .77254 Mabsadd L
.39714 .07521 -0.91448 -0.29713 Mabsadd L
.39714 .07521 -1.46946 -2.02254 Mabsadd L
.39714 .07521 0 -0.96154 Mabsadd L
.39714 .07521 1.46946 -2.02254 Mabsadd L
.39714 .07521 .91448 -0.29713 Mabsadd L
.39714 .07521 2.37764 .77254 Mabsadd L
.39714 .07521 .56518 .7779 Mabsadd L
.39714 .07521 0 2.5 Mabsadd L closepath F
.3981 .07743 0 2.5 Mabsadd m
.3981 .07743 -0.56518 .7779 Mabsadd L
.3981 .07743 -2.37764 .77254 Mabsadd L
.3981 .07743 -0.91448 -0.29713 Mabsadd L
.3981 .07743 -1.46946 -2.02254 Mabsadd L
.3981 .07743 0 -0.96154 Mabsadd L
.3981 .07743 1.46946 -2.02254 Mabsadd L
.3981 .07743 .91448 -0.29713 Mabsadd L
.3981 .07743 2.37764 .77254 Mabsadd L
.3981 .07743 .56518 .7779 Mabsadd L
.3981 .07743 0 2.5 Mabsadd L closepath F
.39905 .08112 0 2.5 Mabsadd m
.39905 .08112 -0.56518 .7779 Mabsadd L
.39905 .08112 -2.37764 .77254 Mabsadd L
.39905 .08112 -0.91448 -0.29713 Mabsadd L
.39905 .08112 -1.46946 -2.02254 Mabsadd L
.39905 .08112 0 -0.96154 Mabsadd L
.39905 .08112 1.46946 -2.02254 Mabsadd L
.39905 .08112 .91448 -0.29713 Mabsadd L
.39905 .08112 2.37764 .77254 Mabsadd L
.39905 .08112 .56518 .7779 Mabsadd L
.39905 .08112 0 2.5 Mabsadd L closepath F
.4 .08111 0 2.5 Mabsadd m
.4 .08111 -0.56518 .7779 Mabsadd L
.4 .08111 -2.37764 .77254 Mabsadd L
.4 .08111 -0.91448 -0.29713 Mabsadd L
.4 .08111 -1.46946 -2.02254 Mabsadd L
.4 .08111 0 -0.96154 Mabsadd L
.4 .08111 1.46946 -2.02254 Mabsadd L
.4 .08111 .91448 -0.29713 Mabsadd L
.4 .08111 2.37764 .77254 Mabsadd L
.4 .08111 .56518 .7779 Mabsadd L
.4 .08111 0 2.5 Mabsadd L closepath F
.40095 .14637 0 2.5 Mabsadd m
.40095 .14637 -0.56518 .7779 Mabsadd L
.40095 .14637 -2.37764 .77254 Mabsadd L
.40095 .14637 -0.91448 -0.29713 Mabsadd L
.40095 .14637 -1.46946 -2.02254 Mabsadd L
.40095 .14637 0 -0.96154 Mabsadd L
.40095 .14637 1.46946 -2.02254 Mabsadd L
.40095 .14637 .91448 -0.29713 Mabsadd L
.40095 .14637 2.37764 .77254 Mabsadd L
.40095 .14637 .56518 .7779 Mabsadd L
.40095 .14637 0 2.5 Mabsadd L closepath F
.4019 .1625 0 2.5 Mabsadd m
.4019 .1625 -0.56518 .7779 Mabsadd L
.4019 .1625 -2.37764 .77254 Mabsadd L
.4019 .1625 -0.91448 -0.29713 Mabsadd L
.4019 .1625 -1.46946 -2.02254 Mabsadd L
.4019 .1625 0 -0.96154 Mabsadd L
.4019 .1625 1.46946 -2.02254 Mabsadd L
.4019 .1625 .91448 -0.29713 Mabsadd L
.4019 .1625 2.37764 .77254 Mabsadd L
.4019 .1625 .56518 .7779 Mabsadd L
.4019 .1625 0 2.5 Mabsadd L closepath F
.40286 .16396 0 2.5 Mabsadd m
.40286 .16396 -0.56518 .7779 Mabsadd L
.40286 .16396 -2.37764 .77254 Mabsadd L
.40286 .16396 -0.91448 -0.29713 Mabsadd L
.40286 .16396 -1.46946 -2.02254 Mabsadd L
.40286 .16396 0 -0.96154 Mabsadd L
.40286 .16396 1.46946 -2.02254 Mabsadd L
.40286 .16396 .91448 -0.29713 Mabsadd L
.40286 .16396 2.37764 .77254 Mabsadd L
.40286 .16396 .56518 .7779 Mabsadd L
.40286 .16396 0 2.5 Mabsadd L closepath F
.40381 .16825 0 2.5 Mabsadd m
.40381 .16825 -0.56518 .7779 Mabsadd L
.40381 .16825 -2.37764 .77254 Mabsadd L
.40381 .16825 -0.91448 -0.29713 Mabsadd L
.40381 .16825 -1.46946 -2.02254 Mabsadd L
.40381 .16825 0 -0.96154 Mabsadd L
.40381 .16825 1.46946 -2.02254 Mabsadd L
.40381 .16825 .91448 -0.29713 Mabsadd L
.40381 .16825 2.37764 .77254 Mabsadd L
.40381 .16825 .56518 .7779 Mabsadd L
.40381 .16825 0 2.5 Mabsadd L closepath F
.40476 .17039 0 2.5 Mabsadd m
.40476 .17039 -0.56518 .7779 Mabsadd L
.40476 .17039 -2.37764 .77254 Mabsadd L
.40476 .17039 -0.91448 -0.29713 Mabsadd L
.40476 .17039 -1.46946 -2.02254 Mabsadd L
.40476 .17039 0 -0.96154 Mabsadd L
.40476 .17039 1.46946 -2.02254 Mabsadd L
.40476 .17039 .91448 -0.29713 Mabsadd L
.40476 .17039 2.37764 .77254 Mabsadd L
.40476 .17039 .56518 .7779 Mabsadd L
.40476 .17039 0 2.5 Mabsadd L closepath F
.40571 .19001 0 2.5 Mabsadd m
.40571 .19001 -0.56518 .7779 Mabsadd L
.40571 .19001 -2.37764 .77254 Mabsadd L
.40571 .19001 -0.91448 -0.29713 Mabsadd L
.40571 .19001 -1.46946 -2.02254 Mabsadd L
.40571 .19001 0 -0.96154 Mabsadd L
.40571 .19001 1.46946 -2.02254 Mabsadd L
.40571 .19001 .91448 -0.29713 Mabsadd L
.40571 .19001 2.37764 .77254 Mabsadd L
.40571 .19001 .56518 .7779 Mabsadd L
.40571 .19001 0 2.5 Mabsadd L closepath F
.40667 .19672 0 2.5 Mabsadd m
.40667 .19672 -0.56518 .7779 Mabsadd L
.40667 .19672 -2.37764 .77254 Mabsadd L
.40667 .19672 -0.91448 -0.29713 Mabsadd L
.40667 .19672 -1.46946 -2.02254 Mabsadd L
.40667 .19672 0 -0.96154 Mabsadd L
.40667 .19672 1.46946 -2.02254 Mabsadd L
.40667 .19672 .91448 -0.29713 Mabsadd L
.40667 .19672 2.37764 .77254 Mabsadd L
.40667 .19672 .56518 .7779 Mabsadd L
.40667 .19672 0 2.5 Mabsadd L closepath F
.40762 .19897 0 2.5 Mabsadd m
.40762 .19897 -0.56518 .7779 Mabsadd L
.40762 .19897 -2.37764 .77254 Mabsadd L
.40762 .19897 -0.91448 -0.29713 Mabsadd L
.40762 .19897 -1.46946 -2.02254 Mabsadd L
.40762 .19897 0 -0.96154 Mabsadd L
.40762 .19897 1.46946 -2.02254 Mabsadd L
.40762 .19897 .91448 -0.29713 Mabsadd L
.40762 .19897 2.37764 .77254 Mabsadd L
.40762 .19897 .56518 .7779 Mabsadd L
.40762 .19897 0 2.5 Mabsadd L closepath F
.40857 .20555 0 2.5 Mabsadd m
.40857 .20555 -0.56518 .7779 Mabsadd L
.40857 .20555 -2.37764 .77254 Mabsadd L
.40857 .20555 -0.91448 -0.29713 Mabsadd L
.40857 .20555 -1.46946 -2.02254 Mabsadd L
.40857 .20555 0 -0.96154 Mabsadd L
.40857 .20555 1.46946 -2.02254 Mabsadd L
.40857 .20555 .91448 -0.29713 Mabsadd L
.40857 .20555 2.37764 .77254 Mabsadd L
.40857 .20555 .56518 .7779 Mabsadd L
.40857 .20555 0 2.5 Mabsadd L closepath F
.40952 .20588 0 2.5 Mabsadd m
.40952 .20588 -0.56518 .7779 Mabsadd L
.40952 .20588 -2.37764 .77254 Mabsadd L
.40952 .20588 -0.91448 -0.29713 Mabsadd L
.40952 .20588 -1.46946 -2.02254 Mabsadd L
.40952 .20588 0 -0.96154 Mabsadd L
.40952 .20588 1.46946 -2.02254 Mabsadd L
.40952 .20588 .91448 -0.29713 Mabsadd L
.40952 .20588 2.37764 .77254 Mabsadd L
.40952 .20588 .56518 .7779 Mabsadd L
.40952 .20588 0 2.5 Mabsadd L closepath F
.41048 .20689 0 2.5 Mabsadd m
.41048 .20689 -0.56518 .7779 Mabsadd L
.41048 .20689 -2.37764 .77254 Mabsadd L
.41048 .20689 -0.91448 -0.29713 Mabsadd L
.41048 .20689 -1.46946 -2.02254 Mabsadd L
.41048 .20689 0 -0.96154 Mabsadd L
.41048 .20689 1.46946 -2.02254 Mabsadd L
.41048 .20689 .91448 -0.29713 Mabsadd L
.41048 .20689 2.37764 .77254 Mabsadd L
.41048 .20689 .56518 .7779 Mabsadd L
.41048 .20689 0 2.5 Mabsadd L closepath F
.41143 .20696 0 2.5 Mabsadd m
.41143 .20696 -0.56518 .7779 Mabsadd L
.41143 .20696 -2.37764 .77254 Mabsadd L
.41143 .20696 -0.91448 -0.29713 Mabsadd L
.41143 .20696 -1.46946 -2.02254 Mabsadd L
.41143 .20696 0 -0.96154 Mabsadd L
.41143 .20696 1.46946 -2.02254 Mabsadd L
.41143 .20696 .91448 -0.29713 Mabsadd L
.41143 .20696 2.37764 .77254 Mabsadd L
.41143 .20696 .56518 .7779 Mabsadd L
.41143 .20696 0 2.5 Mabsadd L closepath F
.41238 .20716 0 2.5 Mabsadd m
.41238 .20716 -0.56518 .7779 Mabsadd L
.41238 .20716 -2.37764 .77254 Mabsadd L
.41238 .20716 -0.91448 -0.29713 Mabsadd L
.41238 .20716 -1.46946 -2.02254 Mabsadd L
.41238 .20716 0 -0.96154 Mabsadd L
.41238 .20716 1.46946 -2.02254 Mabsadd L
.41238 .20716 .91448 -0.29713 Mabsadd L
.41238 .20716 2.37764 .77254 Mabsadd L
.41238 .20716 .56518 .7779 Mabsadd L
.41238 .20716 0 2.5 Mabsadd L closepath F
.41333 .20748 0 2.5 Mabsadd m
.41333 .20748 -0.56518 .7779 Mabsadd L
.41333 .20748 -2.37764 .77254 Mabsadd L
.41333 .20748 -0.91448 -0.29713 Mabsadd L
.41333 .20748 -1.46946 -2.02254 Mabsadd L
.41333 .20748 0 -0.96154 Mabsadd L
.41333 .20748 1.46946 -2.02254 Mabsadd L
.41333 .20748 .91448 -0.29713 Mabsadd L
.41333 .20748 2.37764 .77254 Mabsadd L
.41333 .20748 .56518 .7779 Mabsadd L
.41333 .20748 0 2.5 Mabsadd L closepath F
.41429 .20749 0 2.5 Mabsadd m
.41429 .20749 -0.56518 .7779 Mabsadd L
.41429 .20749 -2.37764 .77254 Mabsadd L
.41429 .20749 -0.91448 -0.29713 Mabsadd L
.41429 .20749 -1.46946 -2.02254 Mabsadd L
.41429 .20749 0 -0.96154 Mabsadd L
.41429 .20749 1.46946 -2.02254 Mabsadd L
.41429 .20749 .91448 -0.29713 Mabsadd L
.41429 .20749 2.37764 .77254 Mabsadd L
.41429 .20749 .56518 .7779 Mabsadd L
.41429 .20749 0 2.5 Mabsadd L closepath F
.41524 .20752 0 2.5 Mabsadd m
.41524 .20752 -0.56518 .7779 Mabsadd L
.41524 .20752 -2.37764 .77254 Mabsadd L
.41524 .20752 -0.91448 -0.29713 Mabsadd L
.41524 .20752 -1.46946 -2.02254 Mabsadd L
.41524 .20752 0 -0.96154 Mabsadd L
.41524 .20752 1.46946 -2.02254 Mabsadd L
.41524 .20752 .91448 -0.29713 Mabsadd L
.41524 .20752 2.37764 .77254 Mabsadd L
.41524 .20752 .56518 .7779 Mabsadd L
.41524 .20752 0 2.5 Mabsadd L closepath F
.41619 .20757 0 2.5 Mabsadd m
.41619 .20757 -0.56518 .7779 Mabsadd L
.41619 .20757 -2.37764 .77254 Mabsadd L
.41619 .20757 -0.91448 -0.29713 Mabsadd L
.41619 .20757 -1.46946 -2.02254 Mabsadd L
.41619 .20757 0 -0.96154 Mabsadd L
.41619 .20757 1.46946 -2.02254 Mabsadd L
.41619 .20757 .91448 -0.29713 Mabsadd L
.41619 .20757 2.37764 .77254 Mabsadd L
.41619 .20757 .56518 .7779 Mabsadd L
.41619 .20757 0 2.5 Mabsadd L closepath F
.41714 .20757 0 2.5 Mabsadd m
.41714 .20757 -0.56518 .7779 Mabsadd L
.41714 .20757 -2.37764 .77254 Mabsadd L
.41714 .20757 -0.91448 -0.29713 Mabsadd L
.41714 .20757 -1.46946 -2.02254 Mabsadd L
.41714 .20757 0 -0.96154 Mabsadd L
.41714 .20757 1.46946 -2.02254 Mabsadd L
.41714 .20757 .91448 -0.29713 Mabsadd L
.41714 .20757 2.37764 .77254 Mabsadd L
.41714 .20757 .56518 .7779 Mabsadd L
.41714 .20757 0 2.5 Mabsadd L closepath F
.4181 .20758 0 2.5 Mabsadd m
.4181 .20758 -0.56518 .7779 Mabsadd L
.4181 .20758 -2.37764 .77254 Mabsadd L
.4181 .20758 -0.91448 -0.29713 Mabsadd L
.4181 .20758 -1.46946 -2.02254 Mabsadd L
.4181 .20758 0 -0.96154 Mabsadd L
.4181 .20758 1.46946 -2.02254 Mabsadd L
.4181 .20758 .91448 -0.29713 Mabsadd L
.4181 .20758 2.37764 .77254 Mabsadd L
.4181 .20758 .56518 .7779 Mabsadd L
.4181 .20758 0 2.5 Mabsadd L closepath F
.41905 .20759 0 2.5 Mabsadd m
.41905 .20759 -0.56518 .7779 Mabsadd L
.41905 .20759 -2.37764 .77254 Mabsadd L
.41905 .20759 -0.91448 -0.29713 Mabsadd L
.41905 .20759 -1.46946 -2.02254 Mabsadd L
.41905 .20759 0 -0.96154 Mabsadd L
.41905 .20759 1.46946 -2.02254 Mabsadd L
.41905 .20759 .91448 -0.29713 Mabsadd L
.41905 .20759 2.37764 .77254 Mabsadd L
.41905 .20759 .56518 .7779 Mabsadd L
.41905 .20759 0 2.5 Mabsadd L closepath F
.42 .20759 0 2.5 Mabsadd m
.42 .20759 -0.56518 .7779 Mabsadd L
.42 .20759 -2.37764 .77254 Mabsadd L
.42 .20759 -0.91448 -0.29713 Mabsadd L
.42 .20759 -1.46946 -2.02254 Mabsadd L
.42 .20759 0 -0.96154 Mabsadd L
.42 .20759 1.46946 -2.02254 Mabsadd L
.42 .20759 .91448 -0.29713 Mabsadd L
.42 .20759 2.37764 .77254 Mabsadd L
.42 .20759 .56518 .7779 Mabsadd L
.42 .20759 0 2.5 Mabsadd L closepath F
.42095 .20759 0 2.5 Mabsadd m
.42095 .20759 -0.56518 .7779 Mabsadd L
.42095 .20759 -2.37764 .77254 Mabsadd L
.42095 .20759 -0.91448 -0.29713 Mabsadd L
.42095 .20759 -1.46946 -2.02254 Mabsadd L
.42095 .20759 0 -0.96154 Mabsadd L
.42095 .20759 1.46946 -2.02254 Mabsadd L
.42095 .20759 .91448 -0.29713 Mabsadd L
.42095 .20759 2.37764 .77254 Mabsadd L
.42095 .20759 .56518 .7779 Mabsadd L
.42095 .20759 0 2.5 Mabsadd L closepath F
.4219 .20759 0 2.5 Mabsadd m
.4219 .20759 -0.56518 .7779 Mabsadd L
.4219 .20759 -2.37764 .77254 Mabsadd L
.4219 .20759 -0.91448 -0.29713 Mabsadd L
.4219 .20759 -1.46946 -2.02254 Mabsadd L
.4219 .20759 0 -0.96154 Mabsadd L
.4219 .20759 1.46946 -2.02254 Mabsadd L
.4219 .20759 .91448 -0.29713 Mabsadd L
.4219 .20759 2.37764 .77254 Mabsadd L
.4219 .20759 .56518 .7779 Mabsadd L
.4219 .20759 0 2.5 Mabsadd L closepath F
.42286 .20759 0 2.5 Mabsadd m
.42286 .20759 -0.56518 .7779 Mabsadd L
.42286 .20759 -2.37764 .77254 Mabsadd L
.42286 .20759 -0.91448 -0.29713 Mabsadd L
.42286 .20759 -1.46946 -2.02254 Mabsadd L
.42286 .20759 0 -0.96154 Mabsadd L
.42286 .20759 1.46946 -2.02254 Mabsadd L
.42286 .20759 .91448 -0.29713 Mabsadd L
.42286 .20759 2.37764 .77254 Mabsadd L
.42286 .20759 .56518 .7779 Mabsadd L
.42286 .20759 0 2.5 Mabsadd L closepath F
.42381 .20759 0 2.5 Mabsadd m
.42381 .20759 -0.56518 .7779 Mabsadd L
.42381 .20759 -2.37764 .77254 Mabsadd L
.42381 .20759 -0.91448 -0.29713 Mabsadd L
.42381 .20759 -1.46946 -2.02254 Mabsadd L
.42381 .20759 0 -0.96154 Mabsadd L
.42381 .20759 1.46946 -2.02254 Mabsadd L
.42381 .20759 .91448 -0.29713 Mabsadd L
.42381 .20759 2.37764 .77254 Mabsadd L
.42381 .20759 .56518 .7779 Mabsadd L
.42381 .20759 0 2.5 Mabsadd L closepath F
.42476 .2076 0 2.5 Mabsadd m
.42476 .2076 -0.56518 .7779 Mabsadd L
.42476 .2076 -2.37764 .77254 Mabsadd L
.42476 .2076 -0.91448 -0.29713 Mabsadd L
.42476 .2076 -1.46946 -2.02254 Mabsadd L
.42476 .2076 0 -0.96154 Mabsadd L
.42476 .2076 1.46946 -2.02254 Mabsadd L
.42476 .2076 .91448 -0.29713 Mabsadd L
.42476 .2076 2.37764 .77254 Mabsadd L
.42476 .2076 .56518 .7779 Mabsadd L
.42476 .2076 0 2.5 Mabsadd L closepath F
.42571 .2076 0 2.5 Mabsadd m
.42571 .2076 -0.56518 .7779 Mabsadd L
.42571 .2076 -2.37764 .77254 Mabsadd L
.42571 .2076 -0.91448 -0.29713 Mabsadd L
.42571 .2076 -1.46946 -2.02254 Mabsadd L
.42571 .2076 0 -0.96154 Mabsadd L
.42571 .2076 1.46946 -2.02254 Mabsadd L
.42571 .2076 .91448 -0.29713 Mabsadd L
.42571 .2076 2.37764 .77254 Mabsadd L
.42571 .2076 .56518 .7779 Mabsadd L
.42571 .2076 0 2.5 Mabsadd L closepath F
.42667 .20759 0 2.5 Mabsadd m
.42667 .20759 -0.56518 .7779 Mabsadd L
.42667 .20759 -2.37764 .77254 Mabsadd L
.42667 .20759 -0.91448 -0.29713 Mabsadd L
.42667 .20759 -1.46946 -2.02254 Mabsadd L
.42667 .20759 0 -0.96154 Mabsadd L
.42667 .20759 1.46946 -2.02254 Mabsadd L
.42667 .20759 .91448 -0.29713 Mabsadd L
.42667 .20759 2.37764 .77254 Mabsadd L
.42667 .20759 .56518 .7779 Mabsadd L
.42667 .20759 0 2.5 Mabsadd L closepath F
.42762 .20759 0 2.5 Mabsadd m
.42762 .20759 -0.56518 .7779 Mabsadd L
.42762 .20759 -2.37764 .77254 Mabsadd L
.42762 .20759 -0.91448 -0.29713 Mabsadd L
.42762 .20759 -1.46946 -2.02254 Mabsadd L
.42762 .20759 0 -0.96154 Mabsadd L
.42762 .20759 1.46946 -2.02254 Mabsadd L
.42762 .20759 .91448 -0.29713 Mabsadd L
.42762 .20759 2.37764 .77254 Mabsadd L
.42762 .20759 .56518 .7779 Mabsadd L
.42762 .20759 0 2.5 Mabsadd L closepath F
.42857 .20759 0 2.5 Mabsadd m
.42857 .20759 -0.56518 .7779 Mabsadd L
.42857 .20759 -2.37764 .77254 Mabsadd L
.42857 .20759 -0.91448 -0.29713 Mabsadd L
.42857 .20759 -1.46946 -2.02254 Mabsadd L
.42857 .20759 0 -0.96154 Mabsadd L
.42857 .20759 1.46946 -2.02254 Mabsadd L
.42857 .20759 .91448 -0.29713 Mabsadd L
.42857 .20759 2.37764 .77254 Mabsadd L
.42857 .20759 .56518 .7779 Mabsadd L
.42857 .20759 0 2.5 Mabsadd L closepath F
.42952 .20759 0 2.5 Mabsadd m
.42952 .20759 -0.56518 .7779 Mabsadd L
.42952 .20759 -2.37764 .77254 Mabsadd L
.42952 .20759 -0.91448 -0.29713 Mabsadd L
.42952 .20759 -1.46946 -2.02254 Mabsadd L
.42952 .20759 0 -0.96154 Mabsadd L
.42952 .20759 1.46946 -2.02254 Mabsadd L
.42952 .20759 .91448 -0.29713 Mabsadd L
.42952 .20759 2.37764 .77254 Mabsadd L
.42952 .20759 .56518 .7779 Mabsadd L
.42952 .20759 0 2.5 Mabsadd L closepath F
.43048 .20759 0 2.5 Mabsadd m
.43048 .20759 -0.56518 .7779 Mabsadd L
.43048 .20759 -2.37764 .77254 Mabsadd L
.43048 .20759 -0.91448 -0.29713 Mabsadd L
.43048 .20759 -1.46946 -2.02254 Mabsadd L
.43048 .20759 0 -0.96154 Mabsadd L
.43048 .20759 1.46946 -2.02254 Mabsadd L
.43048 .20759 .91448 -0.29713 Mabsadd L
.43048 .20759 2.37764 .77254 Mabsadd L
.43048 .20759 .56518 .7779 Mabsadd L
.43048 .20759 0 2.5 Mabsadd L closepath F
.43143 .20758 0 2.5 Mabsadd m
.43143 .20758 -0.56518 .7779 Mabsadd L
.43143 .20758 -2.37764 .77254 Mabsadd L
.43143 .20758 -0.91448 -0.29713 Mabsadd L
.43143 .20758 -1.46946 -2.02254 Mabsadd L
.43143 .20758 0 -0.96154 Mabsadd L
.43143 .20758 1.46946 -2.02254 Mabsadd L
.43143 .20758 .91448 -0.29713 Mabsadd L
.43143 .20758 2.37764 .77254 Mabsadd L
.43143 .20758 .56518 .7779 Mabsadd L
.43143 .20758 0 2.5 Mabsadd L closepath F
.43238 .20757 0 2.5 Mabsadd m
.43238 .20757 -0.56518 .7779 Mabsadd L
.43238 .20757 -2.37764 .77254 Mabsadd L
.43238 .20757 -0.91448 -0.29713 Mabsadd L
.43238 .20757 -1.46946 -2.02254 Mabsadd L
.43238 .20757 0 -0.96154 Mabsadd L
.43238 .20757 1.46946 -2.02254 Mabsadd L
.43238 .20757 .91448 -0.29713 Mabsadd L
.43238 .20757 2.37764 .77254 Mabsadd L
.43238 .20757 .56518 .7779 Mabsadd L
.43238 .20757 0 2.5 Mabsadd L closepath F
.43333 .20757 0 2.5 Mabsadd m
.43333 .20757 -0.56518 .7779 Mabsadd L
.43333 .20757 -2.37764 .77254 Mabsadd L
.43333 .20757 -0.91448 -0.29713 Mabsadd L
.43333 .20757 -1.46946 -2.02254 Mabsadd L
.43333 .20757 0 -0.96154 Mabsadd L
.43333 .20757 1.46946 -2.02254 Mabsadd L
.43333 .20757 .91448 -0.29713 Mabsadd L
.43333 .20757 2.37764 .77254 Mabsadd L
.43333 .20757 .56518 .7779 Mabsadd L
.43333 .20757 0 2.5 Mabsadd L closepath F
.43429 .20756 0 2.5 Mabsadd m
.43429 .20756 -0.56518 .7779 Mabsadd L
.43429 .20756 -2.37764 .77254 Mabsadd L
.43429 .20756 -0.91448 -0.29713 Mabsadd L
.43429 .20756 -1.46946 -2.02254 Mabsadd L
.43429 .20756 0 -0.96154 Mabsadd L
.43429 .20756 1.46946 -2.02254 Mabsadd L
.43429 .20756 .91448 -0.29713 Mabsadd L
.43429 .20756 2.37764 .77254 Mabsadd L
.43429 .20756 .56518 .7779 Mabsadd L
.43429 .20756 0 2.5 Mabsadd L closepath F
.43524 .20754 0 2.5 Mabsadd m
.43524 .20754 -0.56518 .7779 Mabsadd L
.43524 .20754 -2.37764 .77254 Mabsadd L
.43524 .20754 -0.91448 -0.29713 Mabsadd L
.43524 .20754 -1.46946 -2.02254 Mabsadd L
.43524 .20754 0 -0.96154 Mabsadd L
.43524 .20754 1.46946 -2.02254 Mabsadd L
.43524 .20754 .91448 -0.29713 Mabsadd L
.43524 .20754 2.37764 .77254 Mabsadd L
.43524 .20754 .56518 .7779 Mabsadd L
.43524 .20754 0 2.5 Mabsadd L closepath F
.43619 .20753 0 2.5 Mabsadd m
.43619 .20753 -0.56518 .7779 Mabsadd L
.43619 .20753 -2.37764 .77254 Mabsadd L
.43619 .20753 -0.91448 -0.29713 Mabsadd L
.43619 .20753 -1.46946 -2.02254 Mabsadd L
.43619 .20753 0 -0.96154 Mabsadd L
.43619 .20753 1.46946 -2.02254 Mabsadd L
.43619 .20753 .91448 -0.29713 Mabsadd L
.43619 .20753 2.37764 .77254 Mabsadd L
.43619 .20753 .56518 .7779 Mabsadd L
.43619 .20753 0 2.5 Mabsadd L closepath F
.43714 .2075 0 2.5 Mabsadd m
.43714 .2075 -0.56518 .7779 Mabsadd L
.43714 .2075 -2.37764 .77254 Mabsadd L
.43714 .2075 -0.91448 -0.29713 Mabsadd L
.43714 .2075 -1.46946 -2.02254 Mabsadd L
.43714 .2075 0 -0.96154 Mabsadd L
.43714 .2075 1.46946 -2.02254 Mabsadd L
.43714 .2075 .91448 -0.29713 Mabsadd L
.43714 .2075 2.37764 .77254 Mabsadd L
.43714 .2075 .56518 .7779 Mabsadd L
.43714 .2075 0 2.5 Mabsadd L closepath F
.4381 .20736 0 2.5 Mabsadd m
.4381 .20736 -0.56518 .7779 Mabsadd L
.4381 .20736 -2.37764 .77254 Mabsadd L
.4381 .20736 -0.91448 -0.29713 Mabsadd L
.4381 .20736 -1.46946 -2.02254 Mabsadd L
.4381 .20736 0 -0.96154 Mabsadd L
.4381 .20736 1.46946 -2.02254 Mabsadd L
.4381 .20736 .91448 -0.29713 Mabsadd L
.4381 .20736 2.37764 .77254 Mabsadd L
.4381 .20736 .56518 .7779 Mabsadd L
.4381 .20736 0 2.5 Mabsadd L closepath F
.43905 .20733 0 2.5 Mabsadd m
.43905 .20733 -0.56518 .7779 Mabsadd L
.43905 .20733 -2.37764 .77254 Mabsadd L
.43905 .20733 -0.91448 -0.29713 Mabsadd L
.43905 .20733 -1.46946 -2.02254 Mabsadd L
.43905 .20733 0 -0.96154 Mabsadd L
.43905 .20733 1.46946 -2.02254 Mabsadd L
.43905 .20733 .91448 -0.29713 Mabsadd L
.43905 .20733 2.37764 .77254 Mabsadd L
.43905 .20733 .56518 .7779 Mabsadd L
.43905 .20733 0 2.5 Mabsadd L closepath F
.44 .20706 0 2.5 Mabsadd m
.44 .20706 -0.56518 .7779 Mabsadd L
.44 .20706 -2.37764 .77254 Mabsadd L
.44 .20706 -0.91448 -0.29713 Mabsadd L
.44 .20706 -1.46946 -2.02254 Mabsadd L
.44 .20706 0 -0.96154 Mabsadd L
.44 .20706 1.46946 -2.02254 Mabsadd L
.44 .20706 .91448 -0.29713 Mabsadd L
.44 .20706 2.37764 .77254 Mabsadd L
.44 .20706 .56518 .7779 Mabsadd L
.44 .20706 0 2.5 Mabsadd L closepath F
.44095 .20692 0 2.5 Mabsadd m
.44095 .20692 -0.56518 .7779 Mabsadd L
.44095 .20692 -2.37764 .77254 Mabsadd L
.44095 .20692 -0.91448 -0.29713 Mabsadd L
.44095 .20692 -1.46946 -2.02254 Mabsadd L
.44095 .20692 0 -0.96154 Mabsadd L
.44095 .20692 1.46946 -2.02254 Mabsadd L
.44095 .20692 .91448 -0.29713 Mabsadd L
.44095 .20692 2.37764 .77254 Mabsadd L
.44095 .20692 .56518 .7779 Mabsadd L
.44095 .20692 0 2.5 Mabsadd L closepath F
.4419 .2055 0 2.5 Mabsadd m
.4419 .2055 -0.56518 .7779 Mabsadd L
.4419 .2055 -2.37764 .77254 Mabsadd L
.4419 .2055 -0.91448 -0.29713 Mabsadd L
.4419 .2055 -1.46946 -2.02254 Mabsadd L
.4419 .2055 0 -0.96154 Mabsadd L
.4419 .2055 1.46946 -2.02254 Mabsadd L
.4419 .2055 .91448 -0.29713 Mabsadd L
.4419 .2055 2.37764 .77254 Mabsadd L
.4419 .2055 .56518 .7779 Mabsadd L
.4419 .2055 0 2.5 Mabsadd L closepath F
.44286 .20462 0 2.5 Mabsadd m
.44286 .20462 -0.56518 .7779 Mabsadd L
.44286 .20462 -2.37764 .77254 Mabsadd L
.44286 .20462 -0.91448 -0.29713 Mabsadd L
.44286 .20462 -1.46946 -2.02254 Mabsadd L
.44286 .20462 0 -0.96154 Mabsadd L
.44286 .20462 1.46946 -2.02254 Mabsadd L
.44286 .20462 .91448 -0.29713 Mabsadd L
.44286 .20462 2.37764 .77254 Mabsadd L
.44286 .20462 .56518 .7779 Mabsadd L
.44286 .20462 0 2.5 Mabsadd L closepath F
.44381 .20212 0 2.5 Mabsadd m
.44381 .20212 -0.56518 .7779 Mabsadd L
.44381 .20212 -2.37764 .77254 Mabsadd L
.44381 .20212 -0.91448 -0.29713 Mabsadd L
.44381 .20212 -1.46946 -2.02254 Mabsadd L
.44381 .20212 0 -0.96154 Mabsadd L
.44381 .20212 1.46946 -2.02254 Mabsadd L
.44381 .20212 .91448 -0.29713 Mabsadd L
.44381 .20212 2.37764 .77254 Mabsadd L
.44381 .20212 .56518 .7779 Mabsadd L
.44381 .20212 0 2.5 Mabsadd L closepath F
.44476 .19421 0 2.5 Mabsadd m
.44476 .19421 -0.56518 .7779 Mabsadd L
.44476 .19421 -2.37764 .77254 Mabsadd L
.44476 .19421 -0.91448 -0.29713 Mabsadd L
.44476 .19421 -1.46946 -2.02254 Mabsadd L
.44476 .19421 0 -0.96154 Mabsadd L
.44476 .19421 1.46946 -2.02254 Mabsadd L
.44476 .19421 .91448 -0.29713 Mabsadd L
.44476 .19421 2.37764 .77254 Mabsadd L
.44476 .19421 .56518 .7779 Mabsadd L
.44476 .19421 0 2.5 Mabsadd L closepath F
.44571 .16652 0 2.5 Mabsadd m
.44571 .16652 -0.56518 .7779 Mabsadd L
.44571 .16652 -2.37764 .77254 Mabsadd L
.44571 .16652 -0.91448 -0.29713 Mabsadd L
.44571 .16652 -1.46946 -2.02254 Mabsadd L
.44571 .16652 0 -0.96154 Mabsadd L
.44571 .16652 1.46946 -2.02254 Mabsadd L
.44571 .16652 .91448 -0.29713 Mabsadd L
.44571 .16652 2.37764 .77254 Mabsadd L
.44571 .16652 .56518 .7779 Mabsadd L
.44571 .16652 0 2.5 Mabsadd L closepath F
.44667 .16653 0 2.5 Mabsadd m
.44667 .16653 -0.56518 .7779 Mabsadd L
.44667 .16653 -2.37764 .77254 Mabsadd L
.44667 .16653 -0.91448 -0.29713 Mabsadd L
.44667 .16653 -1.46946 -2.02254 Mabsadd L
.44667 .16653 0 -0.96154 Mabsadd L
.44667 .16653 1.46946 -2.02254 Mabsadd L
.44667 .16653 .91448 -0.29713 Mabsadd L
.44667 .16653 2.37764 .77254 Mabsadd L
.44667 .16653 .56518 .7779 Mabsadd L
.44667 .16653 0 2.5 Mabsadd L closepath F
.44762 .16745 0 2.5 Mabsadd m
.44762 .16745 -0.56518 .7779 Mabsadd L
.44762 .16745 -2.37764 .77254 Mabsadd L
.44762 .16745 -0.91448 -0.29713 Mabsadd L
.44762 .16745 -1.46946 -2.02254 Mabsadd L
.44762 .16745 0 -0.96154 Mabsadd L
.44762 .16745 1.46946 -2.02254 Mabsadd L
.44762 .16745 .91448 -0.29713 Mabsadd L
.44762 .16745 2.37764 .77254 Mabsadd L
.44762 .16745 .56518 .7779 Mabsadd L
.44762 .16745 0 2.5 Mabsadd L closepath F
.44857 .18557 0 2.5 Mabsadd m
.44857 .18557 -0.56518 .7779 Mabsadd L
.44857 .18557 -2.37764 .77254 Mabsadd L
.44857 .18557 -0.91448 -0.29713 Mabsadd L
.44857 .18557 -1.46946 -2.02254 Mabsadd L
.44857 .18557 0 -0.96154 Mabsadd L
.44857 .18557 1.46946 -2.02254 Mabsadd L
.44857 .18557 .91448 -0.29713 Mabsadd L
.44857 .18557 2.37764 .77254 Mabsadd L
.44857 .18557 .56518 .7779 Mabsadd L
.44857 .18557 0 2.5 Mabsadd L closepath F
.44952 .20716 0 2.5 Mabsadd m
.44952 .20716 -0.56518 .7779 Mabsadd L
.44952 .20716 -2.37764 .77254 Mabsadd L
.44952 .20716 -0.91448 -0.29713 Mabsadd L
.44952 .20716 -1.46946 -2.02254 Mabsadd L
.44952 .20716 0 -0.96154 Mabsadd L
.44952 .20716 1.46946 -2.02254 Mabsadd L
.44952 .20716 .91448 -0.29713 Mabsadd L
.44952 .20716 2.37764 .77254 Mabsadd L
.44952 .20716 .56518 .7779 Mabsadd L
.44952 .20716 0 2.5 Mabsadd L closepath F
.45048 .2073 0 2.5 Mabsadd m
.45048 .2073 -0.56518 .7779 Mabsadd L
.45048 .2073 -2.37764 .77254 Mabsadd L
.45048 .2073 -0.91448 -0.29713 Mabsadd L
.45048 .2073 -1.46946 -2.02254 Mabsadd L
.45048 .2073 0 -0.96154 Mabsadd L
.45048 .2073 1.46946 -2.02254 Mabsadd L
.45048 .2073 .91448 -0.29713 Mabsadd L
.45048 .2073 2.37764 .77254 Mabsadd L
.45048 .2073 .56518 .7779 Mabsadd L
.45048 .2073 0 2.5 Mabsadd L closepath F
.45143 .20908 0 2.5 Mabsadd m
.45143 .20908 -0.56518 .7779 Mabsadd L
.45143 .20908 -2.37764 .77254 Mabsadd L
.45143 .20908 -0.91448 -0.29713 Mabsadd L
.45143 .20908 -1.46946 -2.02254 Mabsadd L
.45143 .20908 0 -0.96154 Mabsadd L
.45143 .20908 1.46946 -2.02254 Mabsadd L
.45143 .20908 .91448 -0.29713 Mabsadd L
.45143 .20908 2.37764 .77254 Mabsadd L
.45143 .20908 .56518 .7779 Mabsadd L
.45143 .20908 0 2.5 Mabsadd L closepath F
.45238 .21306 0 2.5 Mabsadd m
.45238 .21306 -0.56518 .7779 Mabsadd L
.45238 .21306 -2.37764 .77254 Mabsadd L
.45238 .21306 -0.91448 -0.29713 Mabsadd L
.45238 .21306 -1.46946 -2.02254 Mabsadd L
.45238 .21306 0 -0.96154 Mabsadd L
.45238 .21306 1.46946 -2.02254 Mabsadd L
.45238 .21306 .91448 -0.29713 Mabsadd L
.45238 .21306 2.37764 .77254 Mabsadd L
.45238 .21306 .56518 .7779 Mabsadd L
.45238 .21306 0 2.5 Mabsadd L closepath F
.45333 .21331 0 2.5 Mabsadd m
.45333 .21331 -0.56518 .7779 Mabsadd L
.45333 .21331 -2.37764 .77254 Mabsadd L
.45333 .21331 -0.91448 -0.29713 Mabsadd L
.45333 .21331 -1.46946 -2.02254 Mabsadd L
.45333 .21331 0 -0.96154 Mabsadd L
.45333 .21331 1.46946 -2.02254 Mabsadd L
.45333 .21331 .91448 -0.29713 Mabsadd L
.45333 .21331 2.37764 .77254 Mabsadd L
.45333 .21331 .56518 .7779 Mabsadd L
.45333 .21331 0 2.5 Mabsadd L closepath F
.45429 .2675 0 2.5 Mabsadd m
.45429 .2675 -0.56518 .7779 Mabsadd L
.45429 .2675 -2.37764 .77254 Mabsadd L
.45429 .2675 -0.91448 -0.29713 Mabsadd L
.45429 .2675 -1.46946 -2.02254 Mabsadd L
.45429 .2675 0 -0.96154 Mabsadd L
.45429 .2675 1.46946 -2.02254 Mabsadd L
.45429 .2675 .91448 -0.29713 Mabsadd L
.45429 .2675 2.37764 .77254 Mabsadd L
.45429 .2675 .56518 .7779 Mabsadd L
.45429 .2675 0 2.5 Mabsadd L closepath F
.45524 .28882 0 2.5 Mabsadd m
.45524 .28882 -0.56518 .7779 Mabsadd L
.45524 .28882 -2.37764 .77254 Mabsadd L
.45524 .28882 -0.91448 -0.29713 Mabsadd L
.45524 .28882 -1.46946 -2.02254 Mabsadd L
.45524 .28882 0 -0.96154 Mabsadd L
.45524 .28882 1.46946 -2.02254 Mabsadd L
.45524 .28882 .91448 -0.29713 Mabsadd L
.45524 .28882 2.37764 .77254 Mabsadd L
.45524 .28882 .56518 .7779 Mabsadd L
.45524 .28882 0 2.5 Mabsadd L closepath F
.45619 .29084 0 2.5 Mabsadd m
.45619 .29084 -0.56518 .7779 Mabsadd L
.45619 .29084 -2.37764 .77254 Mabsadd L
.45619 .29084 -0.91448 -0.29713 Mabsadd L
.45619 .29084 -1.46946 -2.02254 Mabsadd L
.45619 .29084 0 -0.96154 Mabsadd L
.45619 .29084 1.46946 -2.02254 Mabsadd L
.45619 .29084 .91448 -0.29713 Mabsadd L
.45619 .29084 2.37764 .77254 Mabsadd L
.45619 .29084 .56518 .7779 Mabsadd L
.45619 .29084 0 2.5 Mabsadd L closepath F
.45714 .29685 0 2.5 Mabsadd m
.45714 .29685 -0.56518 .7779 Mabsadd L
.45714 .29685 -2.37764 .77254 Mabsadd L
.45714 .29685 -0.91448 -0.29713 Mabsadd L
.45714 .29685 -1.46946 -2.02254 Mabsadd L
.45714 .29685 0 -0.96154 Mabsadd L
.45714 .29685 1.46946 -2.02254 Mabsadd L
.45714 .29685 .91448 -0.29713 Mabsadd L
.45714 .29685 2.37764 .77254 Mabsadd L
.45714 .29685 .56518 .7779 Mabsadd L
.45714 .29685 0 2.5 Mabsadd L closepath F
.4581 .29686 0 2.5 Mabsadd m
.4581 .29686 -0.56518 .7779 Mabsadd L
.4581 .29686 -2.37764 .77254 Mabsadd L
.4581 .29686 -0.91448 -0.29713 Mabsadd L
.4581 .29686 -1.46946 -2.02254 Mabsadd L
.4581 .29686 0 -0.96154 Mabsadd L
.4581 .29686 1.46946 -2.02254 Mabsadd L
.4581 .29686 .91448 -0.29713 Mabsadd L
.4581 .29686 2.37764 .77254 Mabsadd L
.4581 .29686 .56518 .7779 Mabsadd L
.4581 .29686 0 2.5 Mabsadd L closepath F
.45905 .29691 0 2.5 Mabsadd m
.45905 .29691 -0.56518 .7779 Mabsadd L
.45905 .29691 -2.37764 .77254 Mabsadd L
.45905 .29691 -0.91448 -0.29713 Mabsadd L
.45905 .29691 -1.46946 -2.02254 Mabsadd L
.45905 .29691 0 -0.96154 Mabsadd L
.45905 .29691 1.46946 -2.02254 Mabsadd L
.45905 .29691 .91448 -0.29713 Mabsadd L
.45905 .29691 2.37764 .77254 Mabsadd L
.45905 .29691 .56518 .7779 Mabsadd L
.45905 .29691 0 2.5 Mabsadd L closepath F
.46 .29692 0 2.5 Mabsadd m
.46 .29692 -0.56518 .7779 Mabsadd L
.46 .29692 -2.37764 .77254 Mabsadd L
.46 .29692 -0.91448 -0.29713 Mabsadd L
.46 .29692 -1.46946 -2.02254 Mabsadd L
.46 .29692 0 -0.96154 Mabsadd L
.46 .29692 1.46946 -2.02254 Mabsadd L
.46 .29692 .91448 -0.29713 Mabsadd L
.46 .29692 2.37764 .77254 Mabsadd L
.46 .29692 .56518 .7779 Mabsadd L
.46 .29692 0 2.5 Mabsadd L closepath F
.46095 .29689 0 2.5 Mabsadd m
.46095 .29689 -0.56518 .7779 Mabsadd L
.46095 .29689 -2.37764 .77254 Mabsadd L
.46095 .29689 -0.91448 -0.29713 Mabsadd L
.46095 .29689 -1.46946 -2.02254 Mabsadd L
.46095 .29689 0 -0.96154 Mabsadd L
.46095 .29689 1.46946 -2.02254 Mabsadd L
.46095 .29689 .91448 -0.29713 Mabsadd L
.46095 .29689 2.37764 .77254 Mabsadd L
.46095 .29689 .56518 .7779 Mabsadd L
.46095 .29689 0 2.5 Mabsadd L closepath F
.4619 .29684 0 2.5 Mabsadd m
.4619 .29684 -0.56518 .7779 Mabsadd L
.4619 .29684 -2.37764 .77254 Mabsadd L
.4619 .29684 -0.91448 -0.29713 Mabsadd L
.4619 .29684 -1.46946 -2.02254 Mabsadd L
.4619 .29684 0 -0.96154 Mabsadd L
.4619 .29684 1.46946 -2.02254 Mabsadd L
.4619 .29684 .91448 -0.29713 Mabsadd L
.4619 .29684 2.37764 .77254 Mabsadd L
.4619 .29684 .56518 .7779 Mabsadd L
.4619 .29684 0 2.5 Mabsadd L closepath F
.46286 .29677 0 2.5 Mabsadd m
.46286 .29677 -0.56518 .7779 Mabsadd L
.46286 .29677 -2.37764 .77254 Mabsadd L
.46286 .29677 -0.91448 -0.29713 Mabsadd L
.46286 .29677 -1.46946 -2.02254 Mabsadd L
.46286 .29677 0 -0.96154 Mabsadd L
.46286 .29677 1.46946 -2.02254 Mabsadd L
.46286 .29677 .91448 -0.29713 Mabsadd L
.46286 .29677 2.37764 .77254 Mabsadd L
.46286 .29677 .56518 .7779 Mabsadd L
.46286 .29677 0 2.5 Mabsadd L closepath F
.46381 .29671 0 2.5 Mabsadd m
.46381 .29671 -0.56518 .7779 Mabsadd L
.46381 .29671 -2.37764 .77254 Mabsadd L
.46381 .29671 -0.91448 -0.29713 Mabsadd L
.46381 .29671 -1.46946 -2.02254 Mabsadd L
.46381 .29671 0 -0.96154 Mabsadd L
.46381 .29671 1.46946 -2.02254 Mabsadd L
.46381 .29671 .91448 -0.29713 Mabsadd L
.46381 .29671 2.37764 .77254 Mabsadd L
.46381 .29671 .56518 .7779 Mabsadd L
.46381 .29671 0 2.5 Mabsadd L closepath F
.46476 .29666 0 2.5 Mabsadd m
.46476 .29666 -0.56518 .7779 Mabsadd L
.46476 .29666 -2.37764 .77254 Mabsadd L
.46476 .29666 -0.91448 -0.29713 Mabsadd L
.46476 .29666 -1.46946 -2.02254 Mabsadd L
.46476 .29666 0 -0.96154 Mabsadd L
.46476 .29666 1.46946 -2.02254 Mabsadd L
.46476 .29666 .91448 -0.29713 Mabsadd L
.46476 .29666 2.37764 .77254 Mabsadd L
.46476 .29666 .56518 .7779 Mabsadd L
.46476 .29666 0 2.5 Mabsadd L closepath F
.46571 .29661 0 2.5 Mabsadd m
.46571 .29661 -0.56518 .7779 Mabsadd L
.46571 .29661 -2.37764 .77254 Mabsadd L
.46571 .29661 -0.91448 -0.29713 Mabsadd L
.46571 .29661 -1.46946 -2.02254 Mabsadd L
.46571 .29661 0 -0.96154 Mabsadd L
.46571 .29661 1.46946 -2.02254 Mabsadd L
.46571 .29661 .91448 -0.29713 Mabsadd L
.46571 .29661 2.37764 .77254 Mabsadd L
.46571 .29661 .56518 .7779 Mabsadd L
.46571 .29661 0 2.5 Mabsadd L closepath F
.46667 .29657 0 2.5 Mabsadd m
.46667 .29657 -0.56518 .7779 Mabsadd L
.46667 .29657 -2.37764 .77254 Mabsadd L
.46667 .29657 -0.91448 -0.29713 Mabsadd L
.46667 .29657 -1.46946 -2.02254 Mabsadd L
.46667 .29657 0 -0.96154 Mabsadd L
.46667 .29657 1.46946 -2.02254 Mabsadd L
.46667 .29657 .91448 -0.29713 Mabsadd L
.46667 .29657 2.37764 .77254 Mabsadd L
.46667 .29657 .56518 .7779 Mabsadd L
.46667 .29657 0 2.5 Mabsadd L closepath F
.46762 .29654 0 2.5 Mabsadd m
.46762 .29654 -0.56518 .7779 Mabsadd L
.46762 .29654 -2.37764 .77254 Mabsadd L
.46762 .29654 -0.91448 -0.29713 Mabsadd L
.46762 .29654 -1.46946 -2.02254 Mabsadd L
.46762 .29654 0 -0.96154 Mabsadd L
.46762 .29654 1.46946 -2.02254 Mabsadd L
.46762 .29654 .91448 -0.29713 Mabsadd L
.46762 .29654 2.37764 .77254 Mabsadd L
.46762 .29654 .56518 .7779 Mabsadd L
.46762 .29654 0 2.5 Mabsadd L closepath F
.46857 .29654 0 2.5 Mabsadd m
.46857 .29654 -0.56518 .7779 Mabsadd L
.46857 .29654 -2.37764 .77254 Mabsadd L
.46857 .29654 -0.91448 -0.29713 Mabsadd L
.46857 .29654 -1.46946 -2.02254 Mabsadd L
.46857 .29654 0 -0.96154 Mabsadd L
.46857 .29654 1.46946 -2.02254 Mabsadd L
.46857 .29654 .91448 -0.29713 Mabsadd L
.46857 .29654 2.37764 .77254 Mabsadd L
.46857 .29654 .56518 .7779 Mabsadd L
.46857 .29654 0 2.5 Mabsadd L closepath F
.46952 .29656 0 2.5 Mabsadd m
.46952 .29656 -0.56518 .7779 Mabsadd L
.46952 .29656 -2.37764 .77254 Mabsadd L
.46952 .29656 -0.91448 -0.29713 Mabsadd L
.46952 .29656 -1.46946 -2.02254 Mabsadd L
.46952 .29656 0 -0.96154 Mabsadd L
.46952 .29656 1.46946 -2.02254 Mabsadd L
.46952 .29656 .91448 -0.29713 Mabsadd L
.46952 .29656 2.37764 .77254 Mabsadd L
.46952 .29656 .56518 .7779 Mabsadd L
.46952 .29656 0 2.5 Mabsadd L closepath F
.47048 .29659 0 2.5 Mabsadd m
.47048 .29659 -0.56518 .7779 Mabsadd L
.47048 .29659 -2.37764 .77254 Mabsadd L
.47048 .29659 -0.91448 -0.29713 Mabsadd L
.47048 .29659 -1.46946 -2.02254 Mabsadd L
.47048 .29659 0 -0.96154 Mabsadd L
.47048 .29659 1.46946 -2.02254 Mabsadd L
.47048 .29659 .91448 -0.29713 Mabsadd L
.47048 .29659 2.37764 .77254 Mabsadd L
.47048 .29659 .56518 .7779 Mabsadd L
.47048 .29659 0 2.5 Mabsadd L closepath F
.47143 .29659 0 2.5 Mabsadd m
.47143 .29659 -0.56518 .7779 Mabsadd L
.47143 .29659 -2.37764 .77254 Mabsadd L
.47143 .29659 -0.91448 -0.29713 Mabsadd L
.47143 .29659 -1.46946 -2.02254 Mabsadd L
.47143 .29659 0 -0.96154 Mabsadd L
.47143 .29659 1.46946 -2.02254 Mabsadd L
.47143 .29659 .91448 -0.29713 Mabsadd L
.47143 .29659 2.37764 .77254 Mabsadd L
.47143 .29659 .56518 .7779 Mabsadd L
.47143 .29659 0 2.5 Mabsadd L closepath F
.47238 .29659 0 2.5 Mabsadd m
.47238 .29659 -0.56518 .7779 Mabsadd L
.47238 .29659 -2.37764 .77254 Mabsadd L
.47238 .29659 -0.91448 -0.29713 Mabsadd L
.47238 .29659 -1.46946 -2.02254 Mabsadd L
.47238 .29659 0 -0.96154 Mabsadd L
.47238 .29659 1.46946 -2.02254 Mabsadd L
.47238 .29659 .91448 -0.29713 Mabsadd L
.47238 .29659 2.37764 .77254 Mabsadd L
.47238 .29659 .56518 .7779 Mabsadd L
.47238 .29659 0 2.5 Mabsadd L closepath F
.47333 .2966 0 2.5 Mabsadd m
.47333 .2966 -0.56518 .7779 Mabsadd L
.47333 .2966 -2.37764 .77254 Mabsadd L
.47333 .2966 -0.91448 -0.29713 Mabsadd L
.47333 .2966 -1.46946 -2.02254 Mabsadd L
.47333 .2966 0 -0.96154 Mabsadd L
.47333 .2966 1.46946 -2.02254 Mabsadd L
.47333 .2966 .91448 -0.29713 Mabsadd L
.47333 .2966 2.37764 .77254 Mabsadd L
.47333 .2966 .56518 .7779 Mabsadd L
.47333 .2966 0 2.5 Mabsadd L closepath F
.47429 .29661 0 2.5 Mabsadd m
.47429 .29661 -0.56518 .7779 Mabsadd L
.47429 .29661 -2.37764 .77254 Mabsadd L
.47429 .29661 -0.91448 -0.29713 Mabsadd L
.47429 .29661 -1.46946 -2.02254 Mabsadd L
.47429 .29661 0 -0.96154 Mabsadd L
.47429 .29661 1.46946 -2.02254 Mabsadd L
.47429 .29661 .91448 -0.29713 Mabsadd L
.47429 .29661 2.37764 .77254 Mabsadd L
.47429 .29661 .56518 .7779 Mabsadd L
.47429 .29661 0 2.5 Mabsadd L closepath F
.47524 .29661 0 2.5 Mabsadd m
.47524 .29661 -0.56518 .7779 Mabsadd L
.47524 .29661 -2.37764 .77254 Mabsadd L
.47524 .29661 -0.91448 -0.29713 Mabsadd L
.47524 .29661 -1.46946 -2.02254 Mabsadd L
.47524 .29661 0 -0.96154 Mabsadd L
.47524 .29661 1.46946 -2.02254 Mabsadd L
.47524 .29661 .91448 -0.29713 Mabsadd L
.47524 .29661 2.37764 .77254 Mabsadd L
.47524 .29661 .56518 .7779 Mabsadd L
.47524 .29661 0 2.5 Mabsadd L closepath F
.47619 .29661 0 2.5 Mabsadd m
.47619 .29661 -0.56518 .7779 Mabsadd L
.47619 .29661 -2.37764 .77254 Mabsadd L
.47619 .29661 -0.91448 -0.29713 Mabsadd L
.47619 .29661 -1.46946 -2.02254 Mabsadd L
.47619 .29661 0 -0.96154 Mabsadd L
.47619 .29661 1.46946 -2.02254 Mabsadd L
.47619 .29661 .91448 -0.29713 Mabsadd L
.47619 .29661 2.37764 .77254 Mabsadd L
.47619 .29661 .56518 .7779 Mabsadd L
.47619 .29661 0 2.5 Mabsadd L closepath F
.47714 .29661 0 2.5 Mabsadd m
.47714 .29661 -0.56518 .7779 Mabsadd L
.47714 .29661 -2.37764 .77254 Mabsadd L
.47714 .29661 -0.91448 -0.29713 Mabsadd L
.47714 .29661 -1.46946 -2.02254 Mabsadd L
.47714 .29661 0 -0.96154 Mabsadd L
.47714 .29661 1.46946 -2.02254 Mabsadd L
.47714 .29661 .91448 -0.29713 Mabsadd L
.47714 .29661 2.37764 .77254 Mabsadd L
.47714 .29661 .56518 .7779 Mabsadd L
.47714 .29661 0 2.5 Mabsadd L closepath F
.4781 .29661 0 2.5 Mabsadd m
.4781 .29661 -0.56518 .7779 Mabsadd L
.4781 .29661 -2.37764 .77254 Mabsadd L
.4781 .29661 -0.91448 -0.29713 Mabsadd L
.4781 .29661 -1.46946 -2.02254 Mabsadd L
.4781 .29661 0 -0.96154 Mabsadd L
.4781 .29661 1.46946 -2.02254 Mabsadd L
.4781 .29661 .91448 -0.29713 Mabsadd L
.4781 .29661 2.37764 .77254 Mabsadd L
.4781 .29661 .56518 .7779 Mabsadd L
.4781 .29661 0 2.5 Mabsadd L closepath F
.47905 .29662 0 2.5 Mabsadd m
.47905 .29662 -0.56518 .7779 Mabsadd L
.47905 .29662 -2.37764 .77254 Mabsadd L
.47905 .29662 -0.91448 -0.29713 Mabsadd L
.47905 .29662 -1.46946 -2.02254 Mabsadd L
.47905 .29662 0 -0.96154 Mabsadd L
.47905 .29662 1.46946 -2.02254 Mabsadd L
.47905 .29662 .91448 -0.29713 Mabsadd L
.47905 .29662 2.37764 .77254 Mabsadd L
.47905 .29662 .56518 .7779 Mabsadd L
.47905 .29662 0 2.5 Mabsadd L closepath F
.48 .29661 0 2.5 Mabsadd m
.48 .29661 -0.56518 .7779 Mabsadd L
.48 .29661 -2.37764 .77254 Mabsadd L
.48 .29661 -0.91448 -0.29713 Mabsadd L
.48 .29661 -1.46946 -2.02254 Mabsadd L
.48 .29661 0 -0.96154 Mabsadd L
.48 .29661 1.46946 -2.02254 Mabsadd L
.48 .29661 .91448 -0.29713 Mabsadd L
.48 .29661 2.37764 .77254 Mabsadd L
.48 .29661 .56518 .7779 Mabsadd L
.48 .29661 0 2.5 Mabsadd L closepath F
.48095 .29661 0 2.5 Mabsadd m
.48095 .29661 -0.56518 .7779 Mabsadd L
.48095 .29661 -2.37764 .77254 Mabsadd L
.48095 .29661 -0.91448 -0.29713 Mabsadd L
.48095 .29661 -1.46946 -2.02254 Mabsadd L
.48095 .29661 0 -0.96154 Mabsadd L
.48095 .29661 1.46946 -2.02254 Mabsadd L
.48095 .29661 .91448 -0.29713 Mabsadd L
.48095 .29661 2.37764 .77254 Mabsadd L
.48095 .29661 .56518 .7779 Mabsadd L
.48095 .29661 0 2.5 Mabsadd L closepath F
.4819 .29661 0 2.5 Mabsadd m
.4819 .29661 -0.56518 .7779 Mabsadd L
.4819 .29661 -2.37764 .77254 Mabsadd L
.4819 .29661 -0.91448 -0.29713 Mabsadd L
.4819 .29661 -1.46946 -2.02254 Mabsadd L
.4819 .29661 0 -0.96154 Mabsadd L
.4819 .29661 1.46946 -2.02254 Mabsadd L
.4819 .29661 .91448 -0.29713 Mabsadd L
.4819 .29661 2.37764 .77254 Mabsadd L
.4819 .29661 .56518 .7779 Mabsadd L
.4819 .29661 0 2.5 Mabsadd L closepath F
.48286 .29661 0 2.5 Mabsadd m
.48286 .29661 -0.56518 .7779 Mabsadd L
.48286 .29661 -2.37764 .77254 Mabsadd L
.48286 .29661 -0.91448 -0.29713 Mabsadd L
.48286 .29661 -1.46946 -2.02254 Mabsadd L
.48286 .29661 0 -0.96154 Mabsadd L
.48286 .29661 1.46946 -2.02254 Mabsadd L
.48286 .29661 .91448 -0.29713 Mabsadd L
.48286 .29661 2.37764 .77254 Mabsadd L
.48286 .29661 .56518 .7779 Mabsadd L
.48286 .29661 0 2.5 Mabsadd L closepath F
.48381 .29661 0 2.5 Mabsadd m
.48381 .29661 -0.56518 .7779 Mabsadd L
.48381 .29661 -2.37764 .77254 Mabsadd L
.48381 .29661 -0.91448 -0.29713 Mabsadd L
.48381 .29661 -1.46946 -2.02254 Mabsadd L
.48381 .29661 0 -0.96154 Mabsadd L
.48381 .29661 1.46946 -2.02254 Mabsadd L
.48381 .29661 .91448 -0.29713 Mabsadd L
.48381 .29661 2.37764 .77254 Mabsadd L
.48381 .29661 .56518 .7779 Mabsadd L
.48381 .29661 0 2.5 Mabsadd L closepath F
.48476 .2966 0 2.5 Mabsadd m
.48476 .2966 -0.56518 .7779 Mabsadd L
.48476 .2966 -2.37764 .77254 Mabsadd L
.48476 .2966 -0.91448 -0.29713 Mabsadd L
.48476 .2966 -1.46946 -2.02254 Mabsadd L
.48476 .2966 0 -0.96154 Mabsadd L
.48476 .2966 1.46946 -2.02254 Mabsadd L
.48476 .2966 .91448 -0.29713 Mabsadd L
.48476 .2966 2.37764 .77254 Mabsadd L
.48476 .2966 .56518 .7779 Mabsadd L
.48476 .2966 0 2.5 Mabsadd L closepath F
.48571 .2966 0 2.5 Mabsadd m
.48571 .2966 -0.56518 .7779 Mabsadd L
.48571 .2966 -2.37764 .77254 Mabsadd L
.48571 .2966 -0.91448 -0.29713 Mabsadd L
.48571 .2966 -1.46946 -2.02254 Mabsadd L
.48571 .2966 0 -0.96154 Mabsadd L
.48571 .2966 1.46946 -2.02254 Mabsadd L
.48571 .2966 .91448 -0.29713 Mabsadd L
.48571 .2966 2.37764 .77254 Mabsadd L
.48571 .2966 .56518 .7779 Mabsadd L
.48571 .2966 0 2.5 Mabsadd L closepath F
.48667 .2966 0 2.5 Mabsadd m
.48667 .2966 -0.56518 .7779 Mabsadd L
.48667 .2966 -2.37764 .77254 Mabsadd L
.48667 .2966 -0.91448 -0.29713 Mabsadd L
.48667 .2966 -1.46946 -2.02254 Mabsadd L
.48667 .2966 0 -0.96154 Mabsadd L
.48667 .2966 1.46946 -2.02254 Mabsadd L
.48667 .2966 .91448 -0.29713 Mabsadd L
.48667 .2966 2.37764 .77254 Mabsadd L
.48667 .2966 .56518 .7779 Mabsadd L
.48667 .2966 0 2.5 Mabsadd L closepath F
.48762 .29658 0 2.5 Mabsadd m
.48762 .29658 -0.56518 .7779 Mabsadd L
.48762 .29658 -2.37764 .77254 Mabsadd L
.48762 .29658 -0.91448 -0.29713 Mabsadd L
.48762 .29658 -1.46946 -2.02254 Mabsadd L
.48762 .29658 0 -0.96154 Mabsadd L
.48762 .29658 1.46946 -2.02254 Mabsadd L
.48762 .29658 .91448 -0.29713 Mabsadd L
.48762 .29658 2.37764 .77254 Mabsadd L
.48762 .29658 .56518 .7779 Mabsadd L
.48762 .29658 0 2.5 Mabsadd L closepath F
.48857 .29658 0 2.5 Mabsadd m
.48857 .29658 -0.56518 .7779 Mabsadd L
.48857 .29658 -2.37764 .77254 Mabsadd L
.48857 .29658 -0.91448 -0.29713 Mabsadd L
.48857 .29658 -1.46946 -2.02254 Mabsadd L
.48857 .29658 0 -0.96154 Mabsadd L
.48857 .29658 1.46946 -2.02254 Mabsadd L
.48857 .29658 .91448 -0.29713 Mabsadd L
.48857 .29658 2.37764 .77254 Mabsadd L
.48857 .29658 .56518 .7779 Mabsadd L
.48857 .29658 0 2.5 Mabsadd L closepath F
.48952 .29656 0 2.5 Mabsadd m
.48952 .29656 -0.56518 .7779 Mabsadd L
.48952 .29656 -2.37764 .77254 Mabsadd L
.48952 .29656 -0.91448 -0.29713 Mabsadd L
.48952 .29656 -1.46946 -2.02254 Mabsadd L
.48952 .29656 0 -0.96154 Mabsadd L
.48952 .29656 1.46946 -2.02254 Mabsadd L
.48952 .29656 .91448 -0.29713 Mabsadd L
.48952 .29656 2.37764 .77254 Mabsadd L
.48952 .29656 .56518 .7779 Mabsadd L
.48952 .29656 0 2.5 Mabsadd L closepath F
.49048 .29649 0 2.5 Mabsadd m
.49048 .29649 -0.56518 .7779 Mabsadd L
.49048 .29649 -2.37764 .77254 Mabsadd L
.49048 .29649 -0.91448 -0.29713 Mabsadd L
.49048 .29649 -1.46946 -2.02254 Mabsadd L
.49048 .29649 0 -0.96154 Mabsadd L
.49048 .29649 1.46946 -2.02254 Mabsadd L
.49048 .29649 .91448 -0.29713 Mabsadd L
.49048 .29649 2.37764 .77254 Mabsadd L
.49048 .29649 .56518 .7779 Mabsadd L
.49048 .29649 0 2.5 Mabsadd L closepath F
.49143 .29648 0 2.5 Mabsadd m
.49143 .29648 -0.56518 .7779 Mabsadd L
.49143 .29648 -2.37764 .77254 Mabsadd L
.49143 .29648 -0.91448 -0.29713 Mabsadd L
.49143 .29648 -1.46946 -2.02254 Mabsadd L
.49143 .29648 0 -0.96154 Mabsadd L
.49143 .29648 1.46946 -2.02254 Mabsadd L
.49143 .29648 .91448 -0.29713 Mabsadd L
.49143 .29648 2.37764 .77254 Mabsadd L
.49143 .29648 .56518 .7779 Mabsadd L
.49143 .29648 0 2.5 Mabsadd L closepath F
.49238 .29636 0 2.5 Mabsadd m
.49238 .29636 -0.56518 .7779 Mabsadd L
.49238 .29636 -2.37764 .77254 Mabsadd L
.49238 .29636 -0.91448 -0.29713 Mabsadd L
.49238 .29636 -1.46946 -2.02254 Mabsadd L
.49238 .29636 0 -0.96154 Mabsadd L
.49238 .29636 1.46946 -2.02254 Mabsadd L
.49238 .29636 .91448 -0.29713 Mabsadd L
.49238 .29636 2.37764 .77254 Mabsadd L
.49238 .29636 .56518 .7779 Mabsadd L
.49238 .29636 0 2.5 Mabsadd L closepath F
.49333 .2963 0 2.5 Mabsadd m
.49333 .2963 -0.56518 .7779 Mabsadd L
.49333 .2963 -2.37764 .77254 Mabsadd L
.49333 .2963 -0.91448 -0.29713 Mabsadd L
.49333 .2963 -1.46946 -2.02254 Mabsadd L
.49333 .2963 0 -0.96154 Mabsadd L
.49333 .2963 1.46946 -2.02254 Mabsadd L
.49333 .2963 .91448 -0.29713 Mabsadd L
.49333 .2963 2.37764 .77254 Mabsadd L
.49333 .2963 .56518 .7779 Mabsadd L
.49333 .2963 0 2.5 Mabsadd L closepath F
.49429 .29576 0 2.5 Mabsadd m
.49429 .29576 -0.56518 .7779 Mabsadd L
.49429 .29576 -2.37764 .77254 Mabsadd L
.49429 .29576 -0.91448 -0.29713 Mabsadd L
.49429 .29576 -1.46946 -2.02254 Mabsadd L
.49429 .29576 0 -0.96154 Mabsadd L
.49429 .29576 1.46946 -2.02254 Mabsadd L
.49429 .29576 .91448 -0.29713 Mabsadd L
.49429 .29576 2.37764 .77254 Mabsadd L
.49429 .29576 .56518 .7779 Mabsadd L
.49429 .29576 0 2.5 Mabsadd L closepath F
.49524 .29546 0 2.5 Mabsadd m
.49524 .29546 -0.56518 .7779 Mabsadd L
.49524 .29546 -2.37764 .77254 Mabsadd L
.49524 .29546 -0.91448 -0.29713 Mabsadd L
.49524 .29546 -1.46946 -2.02254 Mabsadd L
.49524 .29546 0 -0.96154 Mabsadd L
.49524 .29546 1.46946 -2.02254 Mabsadd L
.49524 .29546 .91448 -0.29713 Mabsadd L
.49524 .29546 2.37764 .77254 Mabsadd L
.49524 .29546 .56518 .7779 Mabsadd L
.49524 .29546 0 2.5 Mabsadd L closepath F
.49619 .29228 0 2.5 Mabsadd m
.49619 .29228 -0.56518 .7779 Mabsadd L
.49619 .29228 -2.37764 .77254 Mabsadd L
.49619 .29228 -0.91448 -0.29713 Mabsadd L
.49619 .29228 -1.46946 -2.02254 Mabsadd L
.49619 .29228 0 -0.96154 Mabsadd L
.49619 .29228 1.46946 -2.02254 Mabsadd L
.49619 .29228 .91448 -0.29713 Mabsadd L
.49619 .29228 2.37764 .77254 Mabsadd L
.49619 .29228 .56518 .7779 Mabsadd L
.49619 .29228 0 2.5 Mabsadd L closepath F
.49714 .29011 0 2.5 Mabsadd m
.49714 .29011 -0.56518 .7779 Mabsadd L
.49714 .29011 -2.37764 .77254 Mabsadd L
.49714 .29011 -0.91448 -0.29713 Mabsadd L
.49714 .29011 -1.46946 -2.02254 Mabsadd L
.49714 .29011 0 -0.96154 Mabsadd L
.49714 .29011 1.46946 -2.02254 Mabsadd L
.49714 .29011 .91448 -0.29713 Mabsadd L
.49714 .29011 2.37764 .77254 Mabsadd L
.49714 .29011 .56518 .7779 Mabsadd L
.49714 .29011 0 2.5 Mabsadd L closepath F
.4981 .28333 0 2.5 Mabsadd m
.4981 .28333 -0.56518 .7779 Mabsadd L
.4981 .28333 -2.37764 .77254 Mabsadd L
.4981 .28333 -0.91448 -0.29713 Mabsadd L
.4981 .28333 -1.46946 -2.02254 Mabsadd L
.4981 .28333 0 -0.96154 Mabsadd L
.4981 .28333 1.46946 -2.02254 Mabsadd L
.4981 .28333 .91448 -0.29713 Mabsadd L
.4981 .28333 2.37764 .77254 Mabsadd L
.4981 .28333 .56518 .7779 Mabsadd L
.4981 .28333 0 2.5 Mabsadd L closepath F
.49905 .25993 0 2.5 Mabsadd m
.49905 .25993 -0.56518 .7779 Mabsadd L
.49905 .25993 -2.37764 .77254 Mabsadd L
.49905 .25993 -0.91448 -0.29713 Mabsadd L
.49905 .25993 -1.46946 -2.02254 Mabsadd L
.49905 .25993 0 -0.96154 Mabsadd L
.49905 .25993 1.46946 -2.02254 Mabsadd L
.49905 .25993 .91448 -0.29713 Mabsadd L
.49905 .25993 2.37764 .77254 Mabsadd L
.49905 .25993 .56518 .7779 Mabsadd L
.49905 .25993 0 2.5 Mabsadd L closepath F
.5 .25994 0 2.5 Mabsadd m
.5 .25994 -0.56518 .7779 Mabsadd L
.5 .25994 -2.37764 .77254 Mabsadd L
.5 .25994 -0.91448 -0.29713 Mabsadd L
.5 .25994 -1.46946 -2.02254 Mabsadd L
.5 .25994 0 -0.96154 Mabsadd L
.5 .25994 1.46946 -2.02254 Mabsadd L
.5 .25994 .91448 -0.29713 Mabsadd L
.5 .25994 2.37764 .77254 Mabsadd L
.5 .25994 .56518 .7779 Mabsadd L
.5 .25994 0 2.5 Mabsadd L closepath F
.50095 .26062 0 2.5 Mabsadd m
.50095 .26062 -0.56518 .7779 Mabsadd L
.50095 .26062 -2.37764 .77254 Mabsadd L
.50095 .26062 -0.91448 -0.29713 Mabsadd L
.50095 .26062 -1.46946 -2.02254 Mabsadd L
.50095 .26062 0 -0.96154 Mabsadd L
.50095 .26062 1.46946 -2.02254 Mabsadd L
.50095 .26062 .91448 -0.29713 Mabsadd L
.50095 .26062 2.37764 .77254 Mabsadd L
.50095 .26062 .56518 .7779 Mabsadd L
.50095 .26062 0 2.5 Mabsadd L closepath F
.5019 .27416 0 2.5 Mabsadd m
.5019 .27416 -0.56518 .7779 Mabsadd L
.5019 .27416 -2.37764 .77254 Mabsadd L
.5019 .27416 -0.91448 -0.29713 Mabsadd L
.5019 .27416 -1.46946 -2.02254 Mabsadd L
.5019 .27416 0 -0.96154 Mabsadd L
.5019 .27416 1.46946 -2.02254 Mabsadd L
.5019 .27416 .91448 -0.29713 Mabsadd L
.5019 .27416 2.37764 .77254 Mabsadd L
.5019 .27416 .56518 .7779 Mabsadd L
.5019 .27416 0 2.5 Mabsadd L closepath F
.50286 .28963 0 2.5 Mabsadd m
.50286 .28963 -0.56518 .7779 Mabsadd L
.50286 .28963 -2.37764 .77254 Mabsadd L
.50286 .28963 -0.91448 -0.29713 Mabsadd L
.50286 .28963 -1.46946 -2.02254 Mabsadd L
.50286 .28963 0 -0.96154 Mabsadd L
.50286 .28963 1.46946 -2.02254 Mabsadd L
.50286 .28963 .91448 -0.29713 Mabsadd L
.50286 .28963 2.37764 .77254 Mabsadd L
.50286 .28963 .56518 .7779 Mabsadd L
.50286 .28963 0 2.5 Mabsadd L closepath F
.50381 .45462 0 2.5 Mabsadd m
.50381 .45462 -0.56518 .7779 Mabsadd L
.50381 .45462 -2.37764 .77254 Mabsadd L
.50381 .45462 -0.91448 -0.29713 Mabsadd L
.50381 .45462 -1.46946 -2.02254 Mabsadd L
.50381 .45462 0 -0.96154 Mabsadd L
.50381 .45462 1.46946 -2.02254 Mabsadd L
.50381 .45462 .91448 -0.29713 Mabsadd L
.50381 .45462 2.37764 .77254 Mabsadd L
.50381 .45462 .56518 .7779 Mabsadd L
.50381 .45462 0 2.5 Mabsadd L closepath F
.50476 .45601 0 2.5 Mabsadd m
.50476 .45601 -0.56518 .7779 Mabsadd L
.50476 .45601 -2.37764 .77254 Mabsadd L
.50476 .45601 -0.91448 -0.29713 Mabsadd L
.50476 .45601 -1.46946 -2.02254 Mabsadd L
.50476 .45601 0 -0.96154 Mabsadd L
.50476 .45601 1.46946 -2.02254 Mabsadd L
.50476 .45601 .91448 -0.29713 Mabsadd L
.50476 .45601 2.37764 .77254 Mabsadd L
.50476 .45601 .56518 .7779 Mabsadd L
.50476 .45601 0 2.5 Mabsadd L closepath F
.50571 .4601 0 2.5 Mabsadd m
.50571 .4601 -0.56518 .7779 Mabsadd L
.50571 .4601 -2.37764 .77254 Mabsadd L
.50571 .4601 -0.91448 -0.29713 Mabsadd L
.50571 .4601 -1.46946 -2.02254 Mabsadd L
.50571 .4601 0 -0.96154 Mabsadd L
.50571 .4601 1.46946 -2.02254 Mabsadd L
.50571 .4601 .91448 -0.29713 Mabsadd L
.50571 .4601 2.37764 .77254 Mabsadd L
.50571 .4601 .56518 .7779 Mabsadd L
.50571 .4601 0 2.5 Mabsadd L closepath F
.50667 .46071 0 2.5 Mabsadd m
.50667 .46071 -0.56518 .7779 Mabsadd L
.50667 .46071 -2.37764 .77254 Mabsadd L
.50667 .46071 -0.91448 -0.29713 Mabsadd L
.50667 .46071 -1.46946 -2.02254 Mabsadd L
.50667 .46071 0 -0.96154 Mabsadd L
.50667 .46071 1.46946 -2.02254 Mabsadd L
.50667 .46071 .91448 -0.29713 Mabsadd L
.50667 .46071 2.37764 .77254 Mabsadd L
.50667 .46071 .56518 .7779 Mabsadd L
.50667 .46071 0 2.5 Mabsadd L closepath F
.50762 .48561 0 2.5 Mabsadd m
.50762 .48561 -0.56518 .7779 Mabsadd L
.50762 .48561 -2.37764 .77254 Mabsadd L
.50762 .48561 -0.91448 -0.29713 Mabsadd L
.50762 .48561 -1.46946 -2.02254 Mabsadd L
.50762 .48561 0 -0.96154 Mabsadd L
.50762 .48561 1.46946 -2.02254 Mabsadd L
.50762 .48561 .91448 -0.29713 Mabsadd L
.50762 .48561 2.37764 .77254 Mabsadd L
.50762 .48561 .56518 .7779 Mabsadd L
.50762 .48561 0 2.5 Mabsadd L closepath F
.50857 .51336 0 2.5 Mabsadd m
.50857 .51336 -0.56518 .7779 Mabsadd L
.50857 .51336 -2.37764 .77254 Mabsadd L
.50857 .51336 -0.91448 -0.29713 Mabsadd L
.50857 .51336 -1.46946 -2.02254 Mabsadd L
.50857 .51336 0 -0.96154 Mabsadd L
.50857 .51336 1.46946 -2.02254 Mabsadd L
.50857 .51336 .91448 -0.29713 Mabsadd L
.50857 .51336 2.37764 .77254 Mabsadd L
.50857 .51336 .56518 .7779 Mabsadd L
.50857 .51336 0 2.5 Mabsadd L closepath F
.50952 .51614 0 2.5 Mabsadd m
.50952 .51614 -0.56518 .7779 Mabsadd L
.50952 .51614 -2.37764 .77254 Mabsadd L
.50952 .51614 -0.91448 -0.29713 Mabsadd L
.50952 .51614 -1.46946 -2.02254 Mabsadd L
.50952 .51614 0 -0.96154 Mabsadd L
.50952 .51614 1.46946 -2.02254 Mabsadd L
.50952 .51614 .91448 -0.29713 Mabsadd L
.50952 .51614 2.37764 .77254 Mabsadd L
.50952 .51614 .56518 .7779 Mabsadd L
.50952 .51614 0 2.5 Mabsadd L closepath F
.51048 .52459 0 2.5 Mabsadd m
.51048 .52459 -0.56518 .7779 Mabsadd L
.51048 .52459 -2.37764 .77254 Mabsadd L
.51048 .52459 -0.91448 -0.29713 Mabsadd L
.51048 .52459 -1.46946 -2.02254 Mabsadd L
.51048 .52459 0 -0.96154 Mabsadd L
.51048 .52459 1.46946 -2.02254 Mabsadd L
.51048 .52459 .91448 -0.29713 Mabsadd L
.51048 .52459 2.37764 .77254 Mabsadd L
.51048 .52459 .56518 .7779 Mabsadd L
.51048 .52459 0 2.5 Mabsadd L closepath F
.51143 .52461 0 2.5 Mabsadd m
.51143 .52461 -0.56518 .7779 Mabsadd L
.51143 .52461 -2.37764 .77254 Mabsadd L
.51143 .52461 -0.91448 -0.29713 Mabsadd L
.51143 .52461 -1.46946 -2.02254 Mabsadd L
.51143 .52461 0 -0.96154 Mabsadd L
.51143 .52461 1.46946 -2.02254 Mabsadd L
.51143 .52461 .91448 -0.29713 Mabsadd L
.51143 .52461 2.37764 .77254 Mabsadd L
.51143 .52461 .56518 .7779 Mabsadd L
.51143 .52461 0 2.5 Mabsadd L closepath F
.51238 .52467 0 2.5 Mabsadd m
.51238 .52467 -0.56518 .7779 Mabsadd L
.51238 .52467 -2.37764 .77254 Mabsadd L
.51238 .52467 -0.91448 -0.29713 Mabsadd L
.51238 .52467 -1.46946 -2.02254 Mabsadd L
.51238 .52467 0 -0.96154 Mabsadd L
.51238 .52467 1.46946 -2.02254 Mabsadd L
.51238 .52467 .91448 -0.29713 Mabsadd L
.51238 .52467 2.37764 .77254 Mabsadd L
.51238 .52467 .56518 .7779 Mabsadd L
.51238 .52467 0 2.5 Mabsadd L closepath F
.51333 .52469 0 2.5 Mabsadd m
.51333 .52469 -0.56518 .7779 Mabsadd L
.51333 .52469 -2.37764 .77254 Mabsadd L
.51333 .52469 -0.91448 -0.29713 Mabsadd L
.51333 .52469 -1.46946 -2.02254 Mabsadd L
.51333 .52469 0 -0.96154 Mabsadd L
.51333 .52469 1.46946 -2.02254 Mabsadd L
.51333 .52469 .91448 -0.29713 Mabsadd L
.51333 .52469 2.37764 .77254 Mabsadd L
.51333 .52469 .56518 .7779 Mabsadd L
.51333 .52469 0 2.5 Mabsadd L closepath F
.51429 .52465 0 2.5 Mabsadd m
.51429 .52465 -0.56518 .7779 Mabsadd L
.51429 .52465 -2.37764 .77254 Mabsadd L
.51429 .52465 -0.91448 -0.29713 Mabsadd L
.51429 .52465 -1.46946 -2.02254 Mabsadd L
.51429 .52465 0 -0.96154 Mabsadd L
.51429 .52465 1.46946 -2.02254 Mabsadd L
.51429 .52465 .91448 -0.29713 Mabsadd L
.51429 .52465 2.37764 .77254 Mabsadd L
.51429 .52465 .56518 .7779 Mabsadd L
.51429 .52465 0 2.5 Mabsadd L closepath F
.51524 .52459 0 2.5 Mabsadd m
.51524 .52459 -0.56518 .7779 Mabsadd L
.51524 .52459 -2.37764 .77254 Mabsadd L
.51524 .52459 -0.91448 -0.29713 Mabsadd L
.51524 .52459 -1.46946 -2.02254 Mabsadd L
.51524 .52459 0 -0.96154 Mabsadd L
.51524 .52459 1.46946 -2.02254 Mabsadd L
.51524 .52459 .91448 -0.29713 Mabsadd L
.51524 .52459 2.37764 .77254 Mabsadd L
.51524 .52459 .56518 .7779 Mabsadd L
.51524 .52459 0 2.5 Mabsadd L closepath F
.51619 .52451 0 2.5 Mabsadd m
.51619 .52451 -0.56518 .7779 Mabsadd L
.51619 .52451 -2.37764 .77254 Mabsadd L
.51619 .52451 -0.91448 -0.29713 Mabsadd L
.51619 .52451 -1.46946 -2.02254 Mabsadd L
.51619 .52451 0 -0.96154 Mabsadd L
.51619 .52451 1.46946 -2.02254 Mabsadd L
.51619 .52451 .91448 -0.29713 Mabsadd L
.51619 .52451 2.37764 .77254 Mabsadd L
.51619 .52451 .56518 .7779 Mabsadd L
.51619 .52451 0 2.5 Mabsadd L closepath F
.51714 .52443 0 2.5 Mabsadd m
.51714 .52443 -0.56518 .7779 Mabsadd L
.51714 .52443 -2.37764 .77254 Mabsadd L
.51714 .52443 -0.91448 -0.29713 Mabsadd L
.51714 .52443 -1.46946 -2.02254 Mabsadd L
.51714 .52443 0 -0.96154 Mabsadd L
.51714 .52443 1.46946 -2.02254 Mabsadd L
.51714 .52443 .91448 -0.29713 Mabsadd L
.51714 .52443 2.37764 .77254 Mabsadd L
.51714 .52443 .56518 .7779 Mabsadd L
.51714 .52443 0 2.5 Mabsadd L closepath F
.5181 .52436 0 2.5 Mabsadd m
.5181 .52436 -0.56518 .7779 Mabsadd L
.5181 .52436 -2.37764 .77254 Mabsadd L
.5181 .52436 -0.91448 -0.29713 Mabsadd L
.5181 .52436 -1.46946 -2.02254 Mabsadd L
.5181 .52436 0 -0.96154 Mabsadd L
.5181 .52436 1.46946 -2.02254 Mabsadd L
.5181 .52436 .91448 -0.29713 Mabsadd L
.5181 .52436 2.37764 .77254 Mabsadd L
.5181 .52436 .56518 .7779 Mabsadd L
.5181 .52436 0 2.5 Mabsadd L closepath F
.51905 .52431 0 2.5 Mabsadd m
.51905 .52431 -0.56518 .7779 Mabsadd L
.51905 .52431 -2.37764 .77254 Mabsadd L
.51905 .52431 -0.91448 -0.29713 Mabsadd L
.51905 .52431 -1.46946 -2.02254 Mabsadd L
.51905 .52431 0 -0.96154 Mabsadd L
.51905 .52431 1.46946 -2.02254 Mabsadd L
.51905 .52431 .91448 -0.29713 Mabsadd L
.51905 .52431 2.37764 .77254 Mabsadd L
.51905 .52431 .56518 .7779 Mabsadd L
.51905 .52431 0 2.5 Mabsadd L closepath F
.52 .52433 0 2.5 Mabsadd m
.52 .52433 -0.56518 .7779 Mabsadd L
.52 .52433 -2.37764 .77254 Mabsadd L
.52 .52433 -0.91448 -0.29713 Mabsadd L
.52 .52433 -1.46946 -2.02254 Mabsadd L
.52 .52433 0 -0.96154 Mabsadd L
.52 .52433 1.46946 -2.02254 Mabsadd L
.52 .52433 .91448 -0.29713 Mabsadd L
.52 .52433 2.37764 .77254 Mabsadd L
.52 .52433 .56518 .7779 Mabsadd L
.52 .52433 0 2.5 Mabsadd L closepath F
.52095 .52439 0 2.5 Mabsadd m
.52095 .52439 -0.56518 .7779 Mabsadd L
.52095 .52439 -2.37764 .77254 Mabsadd L
.52095 .52439 -0.91448 -0.29713 Mabsadd L
.52095 .52439 -1.46946 -2.02254 Mabsadd L
.52095 .52439 0 -0.96154 Mabsadd L
.52095 .52439 1.46946 -2.02254 Mabsadd L
.52095 .52439 .91448 -0.29713 Mabsadd L
.52095 .52439 2.37764 .77254 Mabsadd L
.52095 .52439 .56518 .7779 Mabsadd L
.52095 .52439 0 2.5 Mabsadd L closepath F
.5219 .52449 0 2.5 Mabsadd m
.5219 .52449 -0.56518 .7779 Mabsadd L
.5219 .52449 -2.37764 .77254 Mabsadd L
.5219 .52449 -0.91448 -0.29713 Mabsadd L
.5219 .52449 -1.46946 -2.02254 Mabsadd L
.5219 .52449 0 -0.96154 Mabsadd L
.5219 .52449 1.46946 -2.02254 Mabsadd L
.5219 .52449 .91448 -0.29713 Mabsadd L
.5219 .52449 2.37764 .77254 Mabsadd L
.5219 .52449 .56518 .7779 Mabsadd L
.5219 .52449 0 2.5 Mabsadd L closepath F
.52286 .52449 0 2.5 Mabsadd m
.52286 .52449 -0.56518 .7779 Mabsadd L
.52286 .52449 -2.37764 .77254 Mabsadd L
.52286 .52449 -0.91448 -0.29713 Mabsadd L
.52286 .52449 -1.46946 -2.02254 Mabsadd L
.52286 .52449 0 -0.96154 Mabsadd L
.52286 .52449 1.46946 -2.02254 Mabsadd L
.52286 .52449 .91448 -0.29713 Mabsadd L
.52286 .52449 2.37764 .77254 Mabsadd L
.52286 .52449 .56518 .7779 Mabsadd L
.52286 .52449 0 2.5 Mabsadd L closepath F
.52381 .5245 0 2.5 Mabsadd m
.52381 .5245 -0.56518 .7779 Mabsadd L
.52381 .5245 -2.37764 .77254 Mabsadd L
.52381 .5245 -0.91448 -0.29713 Mabsadd L
.52381 .5245 -1.46946 -2.02254 Mabsadd L
.52381 .5245 0 -0.96154 Mabsadd L
.52381 .5245 1.46946 -2.02254 Mabsadd L
.52381 .5245 .91448 -0.29713 Mabsadd L
.52381 .5245 2.37764 .77254 Mabsadd L
.52381 .5245 .56518 .7779 Mabsadd L
.52381 .5245 0 2.5 Mabsadd L closepath F
.52476 .52452 0 2.5 Mabsadd m
.52476 .52452 -0.56518 .7779 Mabsadd L
.52476 .52452 -2.37764 .77254 Mabsadd L
.52476 .52452 -0.91448 -0.29713 Mabsadd L
.52476 .52452 -1.46946 -2.02254 Mabsadd L
.52476 .52452 0 -0.96154 Mabsadd L
.52476 .52452 1.46946 -2.02254 Mabsadd L
.52476 .52452 .91448 -0.29713 Mabsadd L
.52476 .52452 2.37764 .77254 Mabsadd L
.52476 .52452 .56518 .7779 Mabsadd L
.52476 .52452 0 2.5 Mabsadd L closepath F
.52571 .52452 0 2.5 Mabsadd m
.52571 .52452 -0.56518 .7779 Mabsadd L
.52571 .52452 -2.37764 .77254 Mabsadd L
.52571 .52452 -0.91448 -0.29713 Mabsadd L
.52571 .52452 -1.46946 -2.02254 Mabsadd L
.52571 .52452 0 -0.96154 Mabsadd L
.52571 .52452 1.46946 -2.02254 Mabsadd L
.52571 .52452 .91448 -0.29713 Mabsadd L
.52571 .52452 2.37764 .77254 Mabsadd L
.52571 .52452 .56518 .7779 Mabsadd L
.52571 .52452 0 2.5 Mabsadd L closepath F
.52667 .52453 0 2.5 Mabsadd m
.52667 .52453 -0.56518 .7779 Mabsadd L
.52667 .52453 -2.37764 .77254 Mabsadd L
.52667 .52453 -0.91448 -0.29713 Mabsadd L
.52667 .52453 -1.46946 -2.02254 Mabsadd L
.52667 .52453 0 -0.96154 Mabsadd L
.52667 .52453 1.46946 -2.02254 Mabsadd L
.52667 .52453 .91448 -0.29713 Mabsadd L
.52667 .52453 2.37764 .77254 Mabsadd L
.52667 .52453 .56518 .7779 Mabsadd L
.52667 .52453 0 2.5 Mabsadd L closepath F
.52762 .52453 0 2.5 Mabsadd m
.52762 .52453 -0.56518 .7779 Mabsadd L
.52762 .52453 -2.37764 .77254 Mabsadd L
.52762 .52453 -0.91448 -0.29713 Mabsadd L
.52762 .52453 -1.46946 -2.02254 Mabsadd L
.52762 .52453 0 -0.96154 Mabsadd L
.52762 .52453 1.46946 -2.02254 Mabsadd L
.52762 .52453 .91448 -0.29713 Mabsadd L
.52762 .52453 2.37764 .77254 Mabsadd L
.52762 .52453 .56518 .7779 Mabsadd L
.52762 .52453 0 2.5 Mabsadd L closepath F
.52857 .52454 0 2.5 Mabsadd m
.52857 .52454 -0.56518 .7779 Mabsadd L
.52857 .52454 -2.37764 .77254 Mabsadd L
.52857 .52454 -0.91448 -0.29713 Mabsadd L
.52857 .52454 -1.46946 -2.02254 Mabsadd L
.52857 .52454 0 -0.96154 Mabsadd L
.52857 .52454 1.46946 -2.02254 Mabsadd L
.52857 .52454 .91448 -0.29713 Mabsadd L
.52857 .52454 2.37764 .77254 Mabsadd L
.52857 .52454 .56518 .7779 Mabsadd L
.52857 .52454 0 2.5 Mabsadd L closepath F
.52952 .52454 0 2.5 Mabsadd m
.52952 .52454 -0.56518 .7779 Mabsadd L
.52952 .52454 -2.37764 .77254 Mabsadd L
.52952 .52454 -0.91448 -0.29713 Mabsadd L
.52952 .52454 -1.46946 -2.02254 Mabsadd L
.52952 .52454 0 -0.96154 Mabsadd L
.52952 .52454 1.46946 -2.02254 Mabsadd L
.52952 .52454 .91448 -0.29713 Mabsadd L
.52952 .52454 2.37764 .77254 Mabsadd L
.52952 .52454 .56518 .7779 Mabsadd L
.52952 .52454 0 2.5 Mabsadd L closepath F
.53048 .52454 0 2.5 Mabsadd m
.53048 .52454 -0.56518 .7779 Mabsadd L
.53048 .52454 -2.37764 .77254 Mabsadd L
.53048 .52454 -0.91448 -0.29713 Mabsadd L
.53048 .52454 -1.46946 -2.02254 Mabsadd L
.53048 .52454 0 -0.96154 Mabsadd L
.53048 .52454 1.46946 -2.02254 Mabsadd L
.53048 .52454 .91448 -0.29713 Mabsadd L
.53048 .52454 2.37764 .77254 Mabsadd L
.53048 .52454 .56518 .7779 Mabsadd L
.53048 .52454 0 2.5 Mabsadd L closepath F
.53143 .52454 0 2.5 Mabsadd m
.53143 .52454 -0.56518 .7779 Mabsadd L
.53143 .52454 -2.37764 .77254 Mabsadd L
.53143 .52454 -0.91448 -0.29713 Mabsadd L
.53143 .52454 -1.46946 -2.02254 Mabsadd L
.53143 .52454 0 -0.96154 Mabsadd L
.53143 .52454 1.46946 -2.02254 Mabsadd L
.53143 .52454 .91448 -0.29713 Mabsadd L
.53143 .52454 2.37764 .77254 Mabsadd L
.53143 .52454 .56518 .7779 Mabsadd L
.53143 .52454 0 2.5 Mabsadd L closepath F
.53238 .52454 0 2.5 Mabsadd m
.53238 .52454 -0.56518 .7779 Mabsadd L
.53238 .52454 -2.37764 .77254 Mabsadd L
.53238 .52454 -0.91448 -0.29713 Mabsadd L
.53238 .52454 -1.46946 -2.02254 Mabsadd L
.53238 .52454 0 -0.96154 Mabsadd L
.53238 .52454 1.46946 -2.02254 Mabsadd L
.53238 .52454 .91448 -0.29713 Mabsadd L
.53238 .52454 2.37764 .77254 Mabsadd L
.53238 .52454 .56518 .7779 Mabsadd L
.53238 .52454 0 2.5 Mabsadd L closepath F
.53333 .52454 0 2.5 Mabsadd m
.53333 .52454 -0.56518 .7779 Mabsadd L
.53333 .52454 -2.37764 .77254 Mabsadd L
.53333 .52454 -0.91448 -0.29713 Mabsadd L
.53333 .52454 -1.46946 -2.02254 Mabsadd L
.53333 .52454 0 -0.96154 Mabsadd L
.53333 .52454 1.46946 -2.02254 Mabsadd L
.53333 .52454 .91448 -0.29713 Mabsadd L
.53333 .52454 2.37764 .77254 Mabsadd L
.53333 .52454 .56518 .7779 Mabsadd L
.53333 .52454 0 2.5 Mabsadd L closepath F
.53429 .52454 0 2.5 Mabsadd m
.53429 .52454 -0.56518 .7779 Mabsadd L
.53429 .52454 -2.37764 .77254 Mabsadd L
.53429 .52454 -0.91448 -0.29713 Mabsadd L
.53429 .52454 -1.46946 -2.02254 Mabsadd L
.53429 .52454 0 -0.96154 Mabsadd L
.53429 .52454 1.46946 -2.02254 Mabsadd L
.53429 .52454 .91448 -0.29713 Mabsadd L
.53429 .52454 2.37764 .77254 Mabsadd L
.53429 .52454 .56518 .7779 Mabsadd L
.53429 .52454 0 2.5 Mabsadd L closepath F
.53524 .52454 0 2.5 Mabsadd m
.53524 .52454 -0.56518 .7779 Mabsadd L
.53524 .52454 -2.37764 .77254 Mabsadd L
.53524 .52454 -0.91448 -0.29713 Mabsadd L
.53524 .52454 -1.46946 -2.02254 Mabsadd L
.53524 .52454 0 -0.96154 Mabsadd L
.53524 .52454 1.46946 -2.02254 Mabsadd L
.53524 .52454 .91448 -0.29713 Mabsadd L
.53524 .52454 2.37764 .77254 Mabsadd L
.53524 .52454 .56518 .7779 Mabsadd L
.53524 .52454 0 2.5 Mabsadd L closepath F
.53619 .52454 0 2.5 Mabsadd m
.53619 .52454 -0.56518 .7779 Mabsadd L
.53619 .52454 -2.37764 .77254 Mabsadd L
.53619 .52454 -0.91448 -0.29713 Mabsadd L
.53619 .52454 -1.46946 -2.02254 Mabsadd L
.53619 .52454 0 -0.96154 Mabsadd L
.53619 .52454 1.46946 -2.02254 Mabsadd L
.53619 .52454 .91448 -0.29713 Mabsadd L
.53619 .52454 2.37764 .77254 Mabsadd L
.53619 .52454 .56518 .7779 Mabsadd L
.53619 .52454 0 2.5 Mabsadd L closepath F
.53714 .52454 0 2.5 Mabsadd m
.53714 .52454 -0.56518 .7779 Mabsadd L
.53714 .52454 -2.37764 .77254 Mabsadd L
.53714 .52454 -0.91448 -0.29713 Mabsadd L
.53714 .52454 -1.46946 -2.02254 Mabsadd L
.53714 .52454 0 -0.96154 Mabsadd L
.53714 .52454 1.46946 -2.02254 Mabsadd L
.53714 .52454 .91448 -0.29713 Mabsadd L
.53714 .52454 2.37764 .77254 Mabsadd L
.53714 .52454 .56518 .7779 Mabsadd L
.53714 .52454 0 2.5 Mabsadd L closepath F
.5381 .52453 0 2.5 Mabsadd m
.5381 .52453 -0.56518 .7779 Mabsadd L
.5381 .52453 -2.37764 .77254 Mabsadd L
.5381 .52453 -0.91448 -0.29713 Mabsadd L
.5381 .52453 -1.46946 -2.02254 Mabsadd L
.5381 .52453 0 -0.96154 Mabsadd L
.5381 .52453 1.46946 -2.02254 Mabsadd L
.5381 .52453 .91448 -0.29713 Mabsadd L
.5381 .52453 2.37764 .77254 Mabsadd L
.5381 .52453 .56518 .7779 Mabsadd L
.5381 .52453 0 2.5 Mabsadd L closepath F
.53905 .52452 0 2.5 Mabsadd m
.53905 .52452 -0.56518 .7779 Mabsadd L
.53905 .52452 -2.37764 .77254 Mabsadd L
.53905 .52452 -0.91448 -0.29713 Mabsadd L
.53905 .52452 -1.46946 -2.02254 Mabsadd L
.53905 .52452 0 -0.96154 Mabsadd L
.53905 .52452 1.46946 -2.02254 Mabsadd L
.53905 .52452 .91448 -0.29713 Mabsadd L
.53905 .52452 2.37764 .77254 Mabsadd L
.53905 .52452 .56518 .7779 Mabsadd L
.53905 .52452 0 2.5 Mabsadd L closepath F
.54 .52452 0 2.5 Mabsadd m
.54 .52452 -0.56518 .7779 Mabsadd L
.54 .52452 -2.37764 .77254 Mabsadd L
.54 .52452 -0.91448 -0.29713 Mabsadd L
.54 .52452 -1.46946 -2.02254 Mabsadd L
.54 .52452 0 -0.96154 Mabsadd L
.54 .52452 1.46946 -2.02254 Mabsadd L
.54 .52452 .91448 -0.29713 Mabsadd L
.54 .52452 2.37764 .77254 Mabsadd L
.54 .52452 .56518 .7779 Mabsadd L
.54 .52452 0 2.5 Mabsadd L closepath F
.54095 .52452 0 2.5 Mabsadd m
.54095 .52452 -0.56518 .7779 Mabsadd L
.54095 .52452 -2.37764 .77254 Mabsadd L
.54095 .52452 -0.91448 -0.29713 Mabsadd L
.54095 .52452 -1.46946 -2.02254 Mabsadd L
.54095 .52452 0 -0.96154 Mabsadd L
.54095 .52452 1.46946 -2.02254 Mabsadd L
.54095 .52452 .91448 -0.29713 Mabsadd L
.54095 .52452 2.37764 .77254 Mabsadd L
.54095 .52452 .56518 .7779 Mabsadd L
.54095 .52452 0 2.5 Mabsadd L closepath F
.5419 .5245 0 2.5 Mabsadd m
.5419 .5245 -0.56518 .7779 Mabsadd L
.5419 .5245 -2.37764 .77254 Mabsadd L
.5419 .5245 -0.91448 -0.29713 Mabsadd L
.5419 .5245 -1.46946 -2.02254 Mabsadd L
.5419 .5245 0 -0.96154 Mabsadd L
.5419 .5245 1.46946 -2.02254 Mabsadd L
.5419 .5245 .91448 -0.29713 Mabsadd L
.5419 .5245 2.37764 .77254 Mabsadd L
.5419 .5245 .56518 .7779 Mabsadd L
.5419 .5245 0 2.5 Mabsadd L closepath F
.54286 .52449 0 2.5 Mabsadd m
.54286 .52449 -0.56518 .7779 Mabsadd L
.54286 .52449 -2.37764 .77254 Mabsadd L
.54286 .52449 -0.91448 -0.29713 Mabsadd L
.54286 .52449 -1.46946 -2.02254 Mabsadd L
.54286 .52449 0 -0.96154 Mabsadd L
.54286 .52449 1.46946 -2.02254 Mabsadd L
.54286 .52449 .91448 -0.29713 Mabsadd L
.54286 .52449 2.37764 .77254 Mabsadd L
.54286 .52449 .56518 .7779 Mabsadd L
.54286 .52449 0 2.5 Mabsadd L closepath F
.54381 .52446 0 2.5 Mabsadd m
.54381 .52446 -0.56518 .7779 Mabsadd L
.54381 .52446 -2.37764 .77254 Mabsadd L
.54381 .52446 -0.91448 -0.29713 Mabsadd L
.54381 .52446 -1.46946 -2.02254 Mabsadd L
.54381 .52446 0 -0.96154 Mabsadd L
.54381 .52446 1.46946 -2.02254 Mabsadd L
.54381 .52446 .91448 -0.29713 Mabsadd L
.54381 .52446 2.37764 .77254 Mabsadd L
.54381 .52446 .56518 .7779 Mabsadd L
.54381 .52446 0 2.5 Mabsadd L closepath F
.54476 .52434 0 2.5 Mabsadd m
.54476 .52434 -0.56518 .7779 Mabsadd L
.54476 .52434 -2.37764 .77254 Mabsadd L
.54476 .52434 -0.91448 -0.29713 Mabsadd L
.54476 .52434 -1.46946 -2.02254 Mabsadd L
.54476 .52434 0 -0.96154 Mabsadd L
.54476 .52434 1.46946 -2.02254 Mabsadd L
.54476 .52434 .91448 -0.29713 Mabsadd L
.54476 .52434 2.37764 .77254 Mabsadd L
.54476 .52434 .56518 .7779 Mabsadd L
.54476 .52434 0 2.5 Mabsadd L closepath F
.54571 .52432 0 2.5 Mabsadd m
.54571 .52432 -0.56518 .7779 Mabsadd L
.54571 .52432 -2.37764 .77254 Mabsadd L
.54571 .52432 -0.91448 -0.29713 Mabsadd L
.54571 .52432 -1.46946 -2.02254 Mabsadd L
.54571 .52432 0 -0.96154 Mabsadd L
.54571 .52432 1.46946 -2.02254 Mabsadd L
.54571 .52432 .91448 -0.29713 Mabsadd L
.54571 .52432 2.37764 .77254 Mabsadd L
.54571 .52432 .56518 .7779 Mabsadd L
.54571 .52432 0 2.5 Mabsadd L closepath F
.54667 .52409 0 2.5 Mabsadd m
.54667 .52409 -0.56518 .7779 Mabsadd L
.54667 .52409 -2.37764 .77254 Mabsadd L
.54667 .52409 -0.91448 -0.29713 Mabsadd L
.54667 .52409 -1.46946 -2.02254 Mabsadd L
.54667 .52409 0 -0.96154 Mabsadd L
.54667 .52409 1.46946 -2.02254 Mabsadd L
.54667 .52409 .91448 -0.29713 Mabsadd L
.54667 .52409 2.37764 .77254 Mabsadd L
.54667 .52409 .56518 .7779 Mabsadd L
.54667 .52409 0 2.5 Mabsadd L closepath F
.54762 .52398 0 2.5 Mabsadd m
.54762 .52398 -0.56518 .7779 Mabsadd L
.54762 .52398 -2.37764 .77254 Mabsadd L
.54762 .52398 -0.91448 -0.29713 Mabsadd L
.54762 .52398 -1.46946 -2.02254 Mabsadd L
.54762 .52398 0 -0.96154 Mabsadd L
.54762 .52398 1.46946 -2.02254 Mabsadd L
.54762 .52398 .91448 -0.29713 Mabsadd L
.54762 .52398 2.37764 .77254 Mabsadd L
.54762 .52398 .56518 .7779 Mabsadd L
.54762 .52398 0 2.5 Mabsadd L closepath F
.54857 .52285 0 2.5 Mabsadd m
.54857 .52285 -0.56518 .7779 Mabsadd L
.54857 .52285 -2.37764 .77254 Mabsadd L
.54857 .52285 -0.91448 -0.29713 Mabsadd L
.54857 .52285 -1.46946 -2.02254 Mabsadd L
.54857 .52285 0 -0.96154 Mabsadd L
.54857 .52285 1.46946 -2.02254 Mabsadd L
.54857 .52285 .91448 -0.29713 Mabsadd L
.54857 .52285 2.37764 .77254 Mabsadd L
.54857 .52285 .56518 .7779 Mabsadd L
.54857 .52285 0 2.5 Mabsadd L closepath F
.54952 .52218 0 2.5 Mabsadd m
.54952 .52218 -0.56518 .7779 Mabsadd L
.54952 .52218 -2.37764 .77254 Mabsadd L
.54952 .52218 -0.91448 -0.29713 Mabsadd L
.54952 .52218 -1.46946 -2.02254 Mabsadd L
.54952 .52218 0 -0.96154 Mabsadd L
.54952 .52218 1.46946 -2.02254 Mabsadd L
.54952 .52218 .91448 -0.29713 Mabsadd L
.54952 .52218 2.37764 .77254 Mabsadd L
.54952 .52218 .56518 .7779 Mabsadd L
.54952 .52218 0 2.5 Mabsadd L closepath F
.55048 .52028 0 2.5 Mabsadd m
.55048 .52028 -0.56518 .7779 Mabsadd L
.55048 .52028 -2.37764 .77254 Mabsadd L
.55048 .52028 -0.91448 -0.29713 Mabsadd L
.55048 .52028 -1.46946 -2.02254 Mabsadd L
.55048 .52028 0 -0.96154 Mabsadd L
.55048 .52028 1.46946 -2.02254 Mabsadd L
.55048 .52028 .91448 -0.29713 Mabsadd L
.55048 .52028 2.37764 .77254 Mabsadd L
.55048 .52028 .56518 .7779 Mabsadd L
.55048 .52028 0 2.5 Mabsadd L closepath F
.55143 .51447 0 2.5 Mabsadd m
.55143 .51447 -0.56518 .7779 Mabsadd L
.55143 .51447 -2.37764 .77254 Mabsadd L
.55143 .51447 -0.91448 -0.29713 Mabsadd L
.55143 .51447 -1.46946 -2.02254 Mabsadd L
.55143 .51447 0 -0.96154 Mabsadd L
.55143 .51447 1.46946 -2.02254 Mabsadd L
.55143 .51447 .91448 -0.29713 Mabsadd L
.55143 .51447 2.37764 .77254 Mabsadd L
.55143 .51447 .56518 .7779 Mabsadd L
.55143 .51447 0 2.5 Mabsadd L closepath F
.55238 .49465 0 2.5 Mabsadd m
.55238 .49465 -0.56518 .7779 Mabsadd L
.55238 .49465 -2.37764 .77254 Mabsadd L
.55238 .49465 -0.91448 -0.29713 Mabsadd L
.55238 .49465 -1.46946 -2.02254 Mabsadd L
.55238 .49465 0 -0.96154 Mabsadd L
.55238 .49465 1.46946 -2.02254 Mabsadd L
.55238 .49465 .91448 -0.29713 Mabsadd L
.55238 .49465 2.37764 .77254 Mabsadd L
.55238 .49465 .56518 .7779 Mabsadd L
.55238 .49465 0 2.5 Mabsadd L closepath F
.55333 .49466 0 2.5 Mabsadd m
.55333 .49466 -0.56518 .7779 Mabsadd L
.55333 .49466 -2.37764 .77254 Mabsadd L
.55333 .49466 -0.91448 -0.29713 Mabsadd L
.55333 .49466 -1.46946 -2.02254 Mabsadd L
.55333 .49466 0 -0.96154 Mabsadd L
.55333 .49466 1.46946 -2.02254 Mabsadd L
.55333 .49466 .91448 -0.29713 Mabsadd L
.55333 .49466 2.37764 .77254 Mabsadd L
.55333 .49466 .56518 .7779 Mabsadd L
.55333 .49466 0 2.5 Mabsadd L closepath F
.55429 .49517 0 2.5 Mabsadd m
.55429 .49517 -0.56518 .7779 Mabsadd L
.55429 .49517 -2.37764 .77254 Mabsadd L
.55429 .49517 -0.91448 -0.29713 Mabsadd L
.55429 .49517 -1.46946 -2.02254 Mabsadd L
.55429 .49517 0 -0.96154 Mabsadd L
.55429 .49517 1.46946 -2.02254 Mabsadd L
.55429 .49517 .91448 -0.29713 Mabsadd L
.55429 .49517 2.37764 .77254 Mabsadd L
.55429 .49517 .56518 .7779 Mabsadd L
.55429 .49517 0 2.5 Mabsadd L closepath F
.55524 .50529 0 2.5 Mabsadd m
.55524 .50529 -0.56518 .7779 Mabsadd L
.55524 .50529 -2.37764 .77254 Mabsadd L
.55524 .50529 -0.91448 -0.29713 Mabsadd L
.55524 .50529 -1.46946 -2.02254 Mabsadd L
.55524 .50529 0 -0.96154 Mabsadd L
.55524 .50529 1.46946 -2.02254 Mabsadd L
.55524 .50529 .91448 -0.29713 Mabsadd L
.55524 .50529 2.37764 .77254 Mabsadd L
.55524 .50529 .56518 .7779 Mabsadd L
.55524 .50529 0 2.5 Mabsadd L closepath F
.55619 .51633 0 2.5 Mabsadd m
.55619 .51633 -0.56518 .7779 Mabsadd L
.55619 .51633 -2.37764 .77254 Mabsadd L
.55619 .51633 -0.91448 -0.29713 Mabsadd L
.55619 .51633 -1.46946 -2.02254 Mabsadd L
.55619 .51633 0 -0.96154 Mabsadd L
.55619 .51633 1.46946 -2.02254 Mabsadd L
.55619 .51633 .91448 -0.29713 Mabsadd L
.55619 .51633 2.37764 .77254 Mabsadd L
.55619 .51633 .56518 .7779 Mabsadd L
.55619 .51633 0 2.5 Mabsadd L closepath F
.55714 .51695 0 2.5 Mabsadd m
.55714 .51695 -0.56518 .7779 Mabsadd L
.55714 .51695 -2.37764 .77254 Mabsadd L
.55714 .51695 -0.91448 -0.29713 Mabsadd L
.55714 .51695 -1.46946 -2.02254 Mabsadd L
.55714 .51695 0 -0.96154 Mabsadd L
.55714 .51695 1.46946 -2.02254 Mabsadd L
.55714 .51695 .91448 -0.29713 Mabsadd L
.55714 .51695 2.37764 .77254 Mabsadd L
.55714 .51695 .56518 .7779 Mabsadd L
.55714 .51695 0 2.5 Mabsadd L closepath F
.5581 .518 0 2.5 Mabsadd m
.5581 .518 -0.56518 .7779 Mabsadd L
.5581 .518 -2.37764 .77254 Mabsadd L
.5581 .518 -0.91448 -0.29713 Mabsadd L
.5581 .518 -1.46946 -2.02254 Mabsadd L
.5581 .518 0 -0.96154 Mabsadd L
.5581 .518 1.46946 -2.02254 Mabsadd L
.5581 .518 .91448 -0.29713 Mabsadd L
.5581 .518 2.37764 .77254 Mabsadd L
.5581 .518 .56518 .7779 Mabsadd L
.5581 .518 0 2.5 Mabsadd L closepath F
.55905 .52202 0 2.5 Mabsadd m
.55905 .52202 -0.56518 .7779 Mabsadd L
.55905 .52202 -2.37764 .77254 Mabsadd L
.55905 .52202 -0.91448 -0.29713 Mabsadd L
.55905 .52202 -1.46946 -2.02254 Mabsadd L
.55905 .52202 0 -0.96154 Mabsadd L
.55905 .52202 1.46946 -2.02254 Mabsadd L
.55905 .52202 .91448 -0.29713 Mabsadd L
.55905 .52202 2.37764 .77254 Mabsadd L
.55905 .52202 .56518 .7779 Mabsadd L
.55905 .52202 0 2.5 Mabsadd L closepath F
.56 .52311 0 2.5 Mabsadd m
.56 .52311 -0.56518 .7779 Mabsadd L
.56 .52311 -2.37764 .77254 Mabsadd L
.56 .52311 -0.91448 -0.29713 Mabsadd L
.56 .52311 -1.46946 -2.02254 Mabsadd L
.56 .52311 0 -0.96154 Mabsadd L
.56 .52311 1.46946 -2.02254 Mabsadd L
.56 .52311 .91448 -0.29713 Mabsadd L
.56 .52311 2.37764 .77254 Mabsadd L
.56 .52311 .56518 .7779 Mabsadd L
.56 .52311 0 2.5 Mabsadd L closepath F
.56095 .49118 0 2.5 Mabsadd m
.56095 .49118 -0.56518 .7779 Mabsadd L
.56095 .49118 -2.37764 .77254 Mabsadd L
.56095 .49118 -0.91448 -0.29713 Mabsadd L
.56095 .49118 -1.46946 -2.02254 Mabsadd L
.56095 .49118 0 -0.96154 Mabsadd L
.56095 .49118 1.46946 -2.02254 Mabsadd L
.56095 .49118 .91448 -0.29713 Mabsadd L
.56095 .49118 2.37764 .77254 Mabsadd L
.56095 .49118 .56518 .7779 Mabsadd L
.56095 .49118 0 2.5 Mabsadd L closepath F
.5619 .5266 0 2.5 Mabsadd m
.5619 .5266 -0.56518 .7779 Mabsadd L
.5619 .5266 -2.37764 .77254 Mabsadd L
.5619 .5266 -0.91448 -0.29713 Mabsadd L
.5619 .5266 -1.46946 -2.02254 Mabsadd L
.5619 .5266 0 -0.96154 Mabsadd L
.5619 .5266 1.46946 -2.02254 Mabsadd L
.5619 .5266 .91448 -0.29713 Mabsadd L
.5619 .5266 2.37764 .77254 Mabsadd L
.5619 .5266 .56518 .7779 Mabsadd L
.5619 .5266 0 2.5 Mabsadd L closepath F
.56286 .53043 0 2.5 Mabsadd m
.56286 .53043 -0.56518 .7779 Mabsadd L
.56286 .53043 -2.37764 .77254 Mabsadd L
.56286 .53043 -0.91448 -0.29713 Mabsadd L
.56286 .53043 -1.46946 -2.02254 Mabsadd L
.56286 .53043 0 -0.96154 Mabsadd L
.56286 .53043 1.46946 -2.02254 Mabsadd L
.56286 .53043 .91448 -0.29713 Mabsadd L
.56286 .53043 2.37764 .77254 Mabsadd L
.56286 .53043 .56518 .7779 Mabsadd L
.56286 .53043 0 2.5 Mabsadd L closepath F
.56381 .54233 0 2.5 Mabsadd m
.56381 .54233 -0.56518 .7779 Mabsadd L
.56381 .54233 -2.37764 .77254 Mabsadd L
.56381 .54233 -0.91448 -0.29713 Mabsadd L
.56381 .54233 -1.46946 -2.02254 Mabsadd L
.56381 .54233 0 -0.96154 Mabsadd L
.56381 .54233 1.46946 -2.02254 Mabsadd L
.56381 .54233 .91448 -0.29713 Mabsadd L
.56381 .54233 2.37764 .77254 Mabsadd L
.56381 .54233 .56518 .7779 Mabsadd L
.56381 .54233 0 2.5 Mabsadd L closepath F
.56476 .54236 0 2.5 Mabsadd m
.56476 .54236 -0.56518 .7779 Mabsadd L
.56476 .54236 -2.37764 .77254 Mabsadd L
.56476 .54236 -0.91448 -0.29713 Mabsadd L
.56476 .54236 -1.46946 -2.02254 Mabsadd L
.56476 .54236 0 -0.96154 Mabsadd L
.56476 .54236 1.46946 -2.02254 Mabsadd L
.56476 .54236 .91448 -0.29713 Mabsadd L
.56476 .54236 2.37764 .77254 Mabsadd L
.56476 .54236 .56518 .7779 Mabsadd L
.56476 .54236 0 2.5 Mabsadd L closepath F
.56571 .54244 0 2.5 Mabsadd m
.56571 .54244 -0.56518 .7779 Mabsadd L
.56571 .54244 -2.37764 .77254 Mabsadd L
.56571 .54244 -0.91448 -0.29713 Mabsadd L
.56571 .54244 -1.46946 -2.02254 Mabsadd L
.56571 .54244 0 -0.96154 Mabsadd L
.56571 .54244 1.46946 -2.02254 Mabsadd L
.56571 .54244 .91448 -0.29713 Mabsadd L
.56571 .54244 2.37764 .77254 Mabsadd L
.56571 .54244 .56518 .7779 Mabsadd L
.56571 .54244 0 2.5 Mabsadd L closepath F
.56667 .54247 0 2.5 Mabsadd m
.56667 .54247 -0.56518 .7779 Mabsadd L
.56667 .54247 -2.37764 .77254 Mabsadd L
.56667 .54247 -0.91448 -0.29713 Mabsadd L
.56667 .54247 -1.46946 -2.02254 Mabsadd L
.56667 .54247 0 -0.96154 Mabsadd L
.56667 .54247 1.46946 -2.02254 Mabsadd L
.56667 .54247 .91448 -0.29713 Mabsadd L
.56667 .54247 2.37764 .77254 Mabsadd L
.56667 .54247 .56518 .7779 Mabsadd L
.56667 .54247 0 2.5 Mabsadd L closepath F
.56762 .54242 0 2.5 Mabsadd m
.56762 .54242 -0.56518 .7779 Mabsadd L
.56762 .54242 -2.37764 .77254 Mabsadd L
.56762 .54242 -0.91448 -0.29713 Mabsadd L
.56762 .54242 -1.46946 -2.02254 Mabsadd L
.56762 .54242 0 -0.96154 Mabsadd L
.56762 .54242 1.46946 -2.02254 Mabsadd L
.56762 .54242 .91448 -0.29713 Mabsadd L
.56762 .54242 2.37764 .77254 Mabsadd L
.56762 .54242 .56518 .7779 Mabsadd L
.56762 .54242 0 2.5 Mabsadd L closepath F
.56857 .54234 0 2.5 Mabsadd m
.56857 .54234 -0.56518 .7779 Mabsadd L
.56857 .54234 -2.37764 .77254 Mabsadd L
.56857 .54234 -0.91448 -0.29713 Mabsadd L
.56857 .54234 -1.46946 -2.02254 Mabsadd L
.56857 .54234 0 -0.96154 Mabsadd L
.56857 .54234 1.46946 -2.02254 Mabsadd L
.56857 .54234 .91448 -0.29713 Mabsadd L
.56857 .54234 2.37764 .77254 Mabsadd L
.56857 .54234 .56518 .7779 Mabsadd L
.56857 .54234 0 2.5 Mabsadd L closepath F
.56952 .54224 0 2.5 Mabsadd m
.56952 .54224 -0.56518 .7779 Mabsadd L
.56952 .54224 -2.37764 .77254 Mabsadd L
.56952 .54224 -0.91448 -0.29713 Mabsadd L
.56952 .54224 -1.46946 -2.02254 Mabsadd L
.56952 .54224 0 -0.96154 Mabsadd L
.56952 .54224 1.46946 -2.02254 Mabsadd L
.56952 .54224 .91448 -0.29713 Mabsadd L
.56952 .54224 2.37764 .77254 Mabsadd L
.56952 .54224 .56518 .7779 Mabsadd L
.56952 .54224 0 2.5 Mabsadd L closepath F
.57048 .54215 0 2.5 Mabsadd m
.57048 .54215 -0.56518 .7779 Mabsadd L
.57048 .54215 -2.37764 .77254 Mabsadd L
.57048 .54215 -0.91448 -0.29713 Mabsadd L
.57048 .54215 -1.46946 -2.02254 Mabsadd L
.57048 .54215 0 -0.96154 Mabsadd L
.57048 .54215 1.46946 -2.02254 Mabsadd L
.57048 .54215 .91448 -0.29713 Mabsadd L
.57048 .54215 2.37764 .77254 Mabsadd L
.57048 .54215 .56518 .7779 Mabsadd L
.57048 .54215 0 2.5 Mabsadd L closepath F
.57143 .54223 0 2.5 Mabsadd m
.57143 .54223 -0.56518 .7779 Mabsadd L
.57143 .54223 -2.37764 .77254 Mabsadd L
.57143 .54223 -0.91448 -0.29713 Mabsadd L
.57143 .54223 -1.46946 -2.02254 Mabsadd L
.57143 .54223 0 -0.96154 Mabsadd L
.57143 .54223 1.46946 -2.02254 Mabsadd L
.57143 .54223 .91448 -0.29713 Mabsadd L
.57143 .54223 2.37764 .77254 Mabsadd L
.57143 .54223 .56518 .7779 Mabsadd L
.57143 .54223 0 2.5 Mabsadd L closepath F
.57238 .54251 0 2.5 Mabsadd m
.57238 .54251 -0.56518 .7779 Mabsadd L
.57238 .54251 -2.37764 .77254 Mabsadd L
.57238 .54251 -0.91448 -0.29713 Mabsadd L
.57238 .54251 -1.46946 -2.02254 Mabsadd L
.57238 .54251 0 -0.96154 Mabsadd L
.57238 .54251 1.46946 -2.02254 Mabsadd L
.57238 .54251 .91448 -0.29713 Mabsadd L
.57238 .54251 2.37764 .77254 Mabsadd L
.57238 .54251 .56518 .7779 Mabsadd L
.57238 .54251 0 2.5 Mabsadd L closepath F
.57333 .54253 0 2.5 Mabsadd m
.57333 .54253 -0.56518 .7779 Mabsadd L
.57333 .54253 -2.37764 .77254 Mabsadd L
.57333 .54253 -0.91448 -0.29713 Mabsadd L
.57333 .54253 -1.46946 -2.02254 Mabsadd L
.57333 .54253 0 -0.96154 Mabsadd L
.57333 .54253 1.46946 -2.02254 Mabsadd L
.57333 .54253 .91448 -0.29713 Mabsadd L
.57333 .54253 2.37764 .77254 Mabsadd L
.57333 .54253 .56518 .7779 Mabsadd L
.57333 .54253 0 2.5 Mabsadd L closepath F
.57429 .5426 0 2.5 Mabsadd m
.57429 .5426 -0.56518 .7779 Mabsadd L
.57429 .5426 -2.37764 .77254 Mabsadd L
.57429 .5426 -0.91448 -0.29713 Mabsadd L
.57429 .5426 -1.46946 -2.02254 Mabsadd L
.57429 .5426 0 -0.96154 Mabsadd L
.57429 .5426 1.46946 -2.02254 Mabsadd L
.57429 .5426 .91448 -0.29713 Mabsadd L
.57429 .5426 2.37764 .77254 Mabsadd L
.57429 .5426 .56518 .7779 Mabsadd L
.57429 .5426 0 2.5 Mabsadd L closepath F
.57524 .5426 0 2.5 Mabsadd m
.57524 .5426 -0.56518 .7779 Mabsadd L
.57524 .5426 -2.37764 .77254 Mabsadd L
.57524 .5426 -0.91448 -0.29713 Mabsadd L
.57524 .5426 -1.46946 -2.02254 Mabsadd L
.57524 .5426 0 -0.96154 Mabsadd L
.57524 .5426 1.46946 -2.02254 Mabsadd L
.57524 .5426 .91448 -0.29713 Mabsadd L
.57524 .5426 2.37764 .77254 Mabsadd L
.57524 .5426 .56518 .7779 Mabsadd L
.57524 .5426 0 2.5 Mabsadd L closepath F
.57619 .54262 0 2.5 Mabsadd m
.57619 .54262 -0.56518 .7779 Mabsadd L
.57619 .54262 -2.37764 .77254 Mabsadd L
.57619 .54262 -0.91448 -0.29713 Mabsadd L
.57619 .54262 -1.46946 -2.02254 Mabsadd L
.57619 .54262 0 -0.96154 Mabsadd L
.57619 .54262 1.46946 -2.02254 Mabsadd L
.57619 .54262 .91448 -0.29713 Mabsadd L
.57619 .54262 2.37764 .77254 Mabsadd L
.57619 .54262 .56518 .7779 Mabsadd L
.57619 .54262 0 2.5 Mabsadd L closepath F
.57714 .54266 0 2.5 Mabsadd m
.57714 .54266 -0.56518 .7779 Mabsadd L
.57714 .54266 -2.37764 .77254 Mabsadd L
.57714 .54266 -0.91448 -0.29713 Mabsadd L
.57714 .54266 -1.46946 -2.02254 Mabsadd L
.57714 .54266 0 -0.96154 Mabsadd L
.57714 .54266 1.46946 -2.02254 Mabsadd L
.57714 .54266 .91448 -0.29713 Mabsadd L
.57714 .54266 2.37764 .77254 Mabsadd L
.57714 .54266 .56518 .7779 Mabsadd L
.57714 .54266 0 2.5 Mabsadd L closepath F
.5781 .54266 0 2.5 Mabsadd m
.5781 .54266 -0.56518 .7779 Mabsadd L
.5781 .54266 -2.37764 .77254 Mabsadd L
.5781 .54266 -0.91448 -0.29713 Mabsadd L
.5781 .54266 -1.46946 -2.02254 Mabsadd L
.5781 .54266 0 -0.96154 Mabsadd L
.5781 .54266 1.46946 -2.02254 Mabsadd L
.5781 .54266 .91448 -0.29713 Mabsadd L
.5781 .54266 2.37764 .77254 Mabsadd L
.5781 .54266 .56518 .7779 Mabsadd L
.5781 .54266 0 2.5 Mabsadd L closepath F
.57905 .54266 0 2.5 Mabsadd m
.57905 .54266 -0.56518 .7779 Mabsadd L
.57905 .54266 -2.37764 .77254 Mabsadd L
.57905 .54266 -0.91448 -0.29713 Mabsadd L
.57905 .54266 -1.46946 -2.02254 Mabsadd L
.57905 .54266 0 -0.96154 Mabsadd L
.57905 .54266 1.46946 -2.02254 Mabsadd L
.57905 .54266 .91448 -0.29713 Mabsadd L
.57905 .54266 2.37764 .77254 Mabsadd L
.57905 .54266 .56518 .7779 Mabsadd L
.57905 .54266 0 2.5 Mabsadd L closepath F
.58 .54267 0 2.5 Mabsadd m
.58 .54267 -0.56518 .7779 Mabsadd L
.58 .54267 -2.37764 .77254 Mabsadd L
.58 .54267 -0.91448 -0.29713 Mabsadd L
.58 .54267 -1.46946 -2.02254 Mabsadd L
.58 .54267 0 -0.96154 Mabsadd L
.58 .54267 1.46946 -2.02254 Mabsadd L
.58 .54267 .91448 -0.29713 Mabsadd L
.58 .54267 2.37764 .77254 Mabsadd L
.58 .54267 .56518 .7779 Mabsadd L
.58 .54267 0 2.5 Mabsadd L closepath F
.58095 .54268 0 2.5 Mabsadd m
.58095 .54268 -0.56518 .7779 Mabsadd L
.58095 .54268 -2.37764 .77254 Mabsadd L
.58095 .54268 -0.91448 -0.29713 Mabsadd L
.58095 .54268 -1.46946 -2.02254 Mabsadd L
.58095 .54268 0 -0.96154 Mabsadd L
.58095 .54268 1.46946 -2.02254 Mabsadd L
.58095 .54268 .91448 -0.29713 Mabsadd L
.58095 .54268 2.37764 .77254 Mabsadd L
.58095 .54268 .56518 .7779 Mabsadd L
.58095 .54268 0 2.5 Mabsadd L closepath F
.5819 .54268 0 2.5 Mabsadd m
.5819 .54268 -0.56518 .7779 Mabsadd L
.5819 .54268 -2.37764 .77254 Mabsadd L
.5819 .54268 -0.91448 -0.29713 Mabsadd L
.5819 .54268 -1.46946 -2.02254 Mabsadd L
.5819 .54268 0 -0.96154 Mabsadd L
.5819 .54268 1.46946 -2.02254 Mabsadd L
.5819 .54268 .91448 -0.29713 Mabsadd L
.5819 .54268 2.37764 .77254 Mabsadd L
.5819 .54268 .56518 .7779 Mabsadd L
.5819 .54268 0 2.5 Mabsadd L closepath F
.58286 .54268 0 2.5 Mabsadd m
.58286 .54268 -0.56518 .7779 Mabsadd L
.58286 .54268 -2.37764 .77254 Mabsadd L
.58286 .54268 -0.91448 -0.29713 Mabsadd L
.58286 .54268 -1.46946 -2.02254 Mabsadd L
.58286 .54268 0 -0.96154 Mabsadd L
.58286 .54268 1.46946 -2.02254 Mabsadd L
.58286 .54268 .91448 -0.29713 Mabsadd L
.58286 .54268 2.37764 .77254 Mabsadd L
.58286 .54268 .56518 .7779 Mabsadd L
.58286 .54268 0 2.5 Mabsadd L closepath F
.58381 .54268 0 2.5 Mabsadd m
.58381 .54268 -0.56518 .7779 Mabsadd L
.58381 .54268 -2.37764 .77254 Mabsadd L
.58381 .54268 -0.91448 -0.29713 Mabsadd L
.58381 .54268 -1.46946 -2.02254 Mabsadd L
.58381 .54268 0 -0.96154 Mabsadd L
.58381 .54268 1.46946 -2.02254 Mabsadd L
.58381 .54268 .91448 -0.29713 Mabsadd L
.58381 .54268 2.37764 .77254 Mabsadd L
.58381 .54268 .56518 .7779 Mabsadd L
.58381 .54268 0 2.5 Mabsadd L closepath F
.58476 .54268 0 2.5 Mabsadd m
.58476 .54268 -0.56518 .7779 Mabsadd L
.58476 .54268 -2.37764 .77254 Mabsadd L
.58476 .54268 -0.91448 -0.29713 Mabsadd L
.58476 .54268 -1.46946 -2.02254 Mabsadd L
.58476 .54268 0 -0.96154 Mabsadd L
.58476 .54268 1.46946 -2.02254 Mabsadd L
.58476 .54268 .91448 -0.29713 Mabsadd L
.58476 .54268 2.37764 .77254 Mabsadd L
.58476 .54268 .56518 .7779 Mabsadd L
.58476 .54268 0 2.5 Mabsadd L closepath F
.58571 .54268 0 2.5 Mabsadd m
.58571 .54268 -0.56518 .7779 Mabsadd L
.58571 .54268 -2.37764 .77254 Mabsadd L
.58571 .54268 -0.91448 -0.29713 Mabsadd L
.58571 .54268 -1.46946 -2.02254 Mabsadd L
.58571 .54268 0 -0.96154 Mabsadd L
.58571 .54268 1.46946 -2.02254 Mabsadd L
.58571 .54268 .91448 -0.29713 Mabsadd L
.58571 .54268 2.37764 .77254 Mabsadd L
.58571 .54268 .56518 .7779 Mabsadd L
.58571 .54268 0 2.5 Mabsadd L closepath F
.58667 .54268 0 2.5 Mabsadd m
.58667 .54268 -0.56518 .7779 Mabsadd L
.58667 .54268 -2.37764 .77254 Mabsadd L
.58667 .54268 -0.91448 -0.29713 Mabsadd L
.58667 .54268 -1.46946 -2.02254 Mabsadd L
.58667 .54268 0 -0.96154 Mabsadd L
.58667 .54268 1.46946 -2.02254 Mabsadd L
.58667 .54268 .91448 -0.29713 Mabsadd L
.58667 .54268 2.37764 .77254 Mabsadd L
.58667 .54268 .56518 .7779 Mabsadd L
.58667 .54268 0 2.5 Mabsadd L closepath F
.58762 .54268 0 2.5 Mabsadd m
.58762 .54268 -0.56518 .7779 Mabsadd L
.58762 .54268 -2.37764 .77254 Mabsadd L
.58762 .54268 -0.91448 -0.29713 Mabsadd L
.58762 .54268 -1.46946 -2.02254 Mabsadd L
.58762 .54268 0 -0.96154 Mabsadd L
.58762 .54268 1.46946 -2.02254 Mabsadd L
.58762 .54268 .91448 -0.29713 Mabsadd L
.58762 .54268 2.37764 .77254 Mabsadd L
.58762 .54268 .56518 .7779 Mabsadd L
.58762 .54268 0 2.5 Mabsadd L closepath F
.58857 .54267 0 2.5 Mabsadd m
.58857 .54267 -0.56518 .7779 Mabsadd L
.58857 .54267 -2.37764 .77254 Mabsadd L
.58857 .54267 -0.91448 -0.29713 Mabsadd L
.58857 .54267 -1.46946 -2.02254 Mabsadd L
.58857 .54267 0 -0.96154 Mabsadd L
.58857 .54267 1.46946 -2.02254 Mabsadd L
.58857 .54267 .91448 -0.29713 Mabsadd L
.58857 .54267 2.37764 .77254 Mabsadd L
.58857 .54267 .56518 .7779 Mabsadd L
.58857 .54267 0 2.5 Mabsadd L closepath F
.58952 .54267 0 2.5 Mabsadd m
.58952 .54267 -0.56518 .7779 Mabsadd L
.58952 .54267 -2.37764 .77254 Mabsadd L
.58952 .54267 -0.91448 -0.29713 Mabsadd L
.58952 .54267 -1.46946 -2.02254 Mabsadd L
.58952 .54267 0 -0.96154 Mabsadd L
.58952 .54267 1.46946 -2.02254 Mabsadd L
.58952 .54267 .91448 -0.29713 Mabsadd L
.58952 .54267 2.37764 .77254 Mabsadd L
.58952 .54267 .56518 .7779 Mabsadd L
.58952 .54267 0 2.5 Mabsadd L closepath F
.59048 .54267 0 2.5 Mabsadd m
.59048 .54267 -0.56518 .7779 Mabsadd L
.59048 .54267 -2.37764 .77254 Mabsadd L
.59048 .54267 -0.91448 -0.29713 Mabsadd L
.59048 .54267 -1.46946 -2.02254 Mabsadd L
.59048 .54267 0 -0.96154 Mabsadd L
.59048 .54267 1.46946 -2.02254 Mabsadd L
.59048 .54267 .91448 -0.29713 Mabsadd L
.59048 .54267 2.37764 .77254 Mabsadd L
.59048 .54267 .56518 .7779 Mabsadd L
.59048 .54267 0 2.5 Mabsadd L closepath F
.59143 .54267 0 2.5 Mabsadd m
.59143 .54267 -0.56518 .7779 Mabsadd L
.59143 .54267 -2.37764 .77254 Mabsadd L
.59143 .54267 -0.91448 -0.29713 Mabsadd L
.59143 .54267 -1.46946 -2.02254 Mabsadd L
.59143 .54267 0 -0.96154 Mabsadd L
.59143 .54267 1.46946 -2.02254 Mabsadd L
.59143 .54267 .91448 -0.29713 Mabsadd L
.59143 .54267 2.37764 .77254 Mabsadd L
.59143 .54267 .56518 .7779 Mabsadd L
.59143 .54267 0 2.5 Mabsadd L closepath F
.59238 .54266 0 2.5 Mabsadd m
.59238 .54266 -0.56518 .7779 Mabsadd L
.59238 .54266 -2.37764 .77254 Mabsadd L
.59238 .54266 -0.91448 -0.29713 Mabsadd L
.59238 .54266 -1.46946 -2.02254 Mabsadd L
.59238 .54266 0 -0.96154 Mabsadd L
.59238 .54266 1.46946 -2.02254 Mabsadd L
.59238 .54266 .91448 -0.29713 Mabsadd L
.59238 .54266 2.37764 .77254 Mabsadd L
.59238 .54266 .56518 .7779 Mabsadd L
.59238 .54266 0 2.5 Mabsadd L closepath F
.59333 .54266 0 2.5 Mabsadd m
.59333 .54266 -0.56518 .7779 Mabsadd L
.59333 .54266 -2.37764 .77254 Mabsadd L
.59333 .54266 -0.91448 -0.29713 Mabsadd L
.59333 .54266 -1.46946 -2.02254 Mabsadd L
.59333 .54266 0 -0.96154 Mabsadd L
.59333 .54266 1.46946 -2.02254 Mabsadd L
.59333 .54266 .91448 -0.29713 Mabsadd L
.59333 .54266 2.37764 .77254 Mabsadd L
.59333 .54266 .56518 .7779 Mabsadd L
.59333 .54266 0 2.5 Mabsadd L closepath F
.59429 .54265 0 2.5 Mabsadd m
.59429 .54265 -0.56518 .7779 Mabsadd L
.59429 .54265 -2.37764 .77254 Mabsadd L
.59429 .54265 -0.91448 -0.29713 Mabsadd L
.59429 .54265 -1.46946 -2.02254 Mabsadd L
.59429 .54265 0 -0.96154 Mabsadd L
.59429 .54265 1.46946 -2.02254 Mabsadd L
.59429 .54265 .91448 -0.29713 Mabsadd L
.59429 .54265 2.37764 .77254 Mabsadd L
.59429 .54265 .56518 .7779 Mabsadd L
.59429 .54265 0 2.5 Mabsadd L closepath F
.59524 .54263 0 2.5 Mabsadd m
.59524 .54263 -0.56518 .7779 Mabsadd L
.59524 .54263 -2.37764 .77254 Mabsadd L
.59524 .54263 -0.91448 -0.29713 Mabsadd L
.59524 .54263 -1.46946 -2.02254 Mabsadd L
.59524 .54263 0 -0.96154 Mabsadd L
.59524 .54263 1.46946 -2.02254 Mabsadd L
.59524 .54263 .91448 -0.29713 Mabsadd L
.59524 .54263 2.37764 .77254 Mabsadd L
.59524 .54263 .56518 .7779 Mabsadd L
.59524 .54263 0 2.5 Mabsadd L closepath F
.59619 .54263 0 2.5 Mabsadd m
.59619 .54263 -0.56518 .7779 Mabsadd L
.59619 .54263 -2.37764 .77254 Mabsadd L
.59619 .54263 -0.91448 -0.29713 Mabsadd L
.59619 .54263 -1.46946 -2.02254 Mabsadd L
.59619 .54263 0 -0.96154 Mabsadd L
.59619 .54263 1.46946 -2.02254 Mabsadd L
.59619 .54263 .91448 -0.29713 Mabsadd L
.59619 .54263 2.37764 .77254 Mabsadd L
.59619 .54263 .56518 .7779 Mabsadd L
.59619 .54263 0 2.5 Mabsadd L closepath F
.59714 .5426 0 2.5 Mabsadd m
.59714 .5426 -0.56518 .7779 Mabsadd L
.59714 .5426 -2.37764 .77254 Mabsadd L
.59714 .5426 -0.91448 -0.29713 Mabsadd L
.59714 .5426 -1.46946 -2.02254 Mabsadd L
.59714 .5426 0 -0.96154 Mabsadd L
.59714 .5426 1.46946 -2.02254 Mabsadd L
.59714 .5426 .91448 -0.29713 Mabsadd L
.59714 .5426 2.37764 .77254 Mabsadd L
.59714 .5426 .56518 .7779 Mabsadd L
.59714 .5426 0 2.5 Mabsadd L closepath F
.5981 .54249 0 2.5 Mabsadd m
.5981 .54249 -0.56518 .7779 Mabsadd L
.5981 .54249 -2.37764 .77254 Mabsadd L
.5981 .54249 -0.91448 -0.29713 Mabsadd L
.5981 .54249 -1.46946 -2.02254 Mabsadd L
.5981 .54249 0 -0.96154 Mabsadd L
.5981 .54249 1.46946 -2.02254 Mabsadd L
.5981 .54249 .91448 -0.29713 Mabsadd L
.5981 .54249 2.37764 .77254 Mabsadd L
.5981 .54249 .56518 .7779 Mabsadd L
.5981 .54249 0 2.5 Mabsadd L closepath F
.59905 .54247 0 2.5 Mabsadd m
.59905 .54247 -0.56518 .7779 Mabsadd L
.59905 .54247 -2.37764 .77254 Mabsadd L
.59905 .54247 -0.91448 -0.29713 Mabsadd L
.59905 .54247 -1.46946 -2.02254 Mabsadd L
.59905 .54247 0 -0.96154 Mabsadd L
.59905 .54247 1.46946 -2.02254 Mabsadd L
.59905 .54247 .91448 -0.29713 Mabsadd L
.59905 .54247 2.37764 .77254 Mabsadd L
.59905 .54247 .56518 .7779 Mabsadd L
.59905 .54247 0 2.5 Mabsadd L closepath F
.6 .54226 0 2.5 Mabsadd m
.6 .54226 -0.56518 .7779 Mabsadd L
.6 .54226 -2.37764 .77254 Mabsadd L
.6 .54226 -0.91448 -0.29713 Mabsadd L
.6 .54226 -1.46946 -2.02254 Mabsadd L
.6 .54226 0 -0.96154 Mabsadd L
.6 .54226 1.46946 -2.02254 Mabsadd L
.6 .54226 .91448 -0.29713 Mabsadd L
.6 .54226 2.37764 .77254 Mabsadd L
.6 .54226 .56518 .7779 Mabsadd L
.6 .54226 0 2.5 Mabsadd L closepath F
.60095 .54216 0 2.5 Mabsadd m
.60095 .54216 -0.56518 .7779 Mabsadd L
.60095 .54216 -2.37764 .77254 Mabsadd L
.60095 .54216 -0.91448 -0.29713 Mabsadd L
.60095 .54216 -1.46946 -2.02254 Mabsadd L
.60095 .54216 0 -0.96154 Mabsadd L
.60095 .54216 1.46946 -2.02254 Mabsadd L
.60095 .54216 .91448 -0.29713 Mabsadd L
.60095 .54216 2.37764 .77254 Mabsadd L
.60095 .54216 .56518 .7779 Mabsadd L
.60095 .54216 0 2.5 Mabsadd L closepath F
.6019 .54116 0 2.5 Mabsadd m
.6019 .54116 -0.56518 .7779 Mabsadd L
.6019 .54116 -2.37764 .77254 Mabsadd L
.6019 .54116 -0.91448 -0.29713 Mabsadd L
.6019 .54116 -1.46946 -2.02254 Mabsadd L
.6019 .54116 0 -0.96154 Mabsadd L
.6019 .54116 1.46946 -2.02254 Mabsadd L
.6019 .54116 .91448 -0.29713 Mabsadd L
.6019 .54116 2.37764 .77254 Mabsadd L
.6019 .54116 .56518 .7779 Mabsadd L
.6019 .54116 0 2.5 Mabsadd L closepath F
.60286 .54056 0 2.5 Mabsadd m
.60286 .54056 -0.56518 .7779 Mabsadd L
.60286 .54056 -2.37764 .77254 Mabsadd L
.60286 .54056 -0.91448 -0.29713 Mabsadd L
.60286 .54056 -1.46946 -2.02254 Mabsadd L
.60286 .54056 0 -0.96154 Mabsadd L
.60286 .54056 1.46946 -2.02254 Mabsadd L
.60286 .54056 .91448 -0.29713 Mabsadd L
.60286 .54056 2.37764 .77254 Mabsadd L
.60286 .54056 .56518 .7779 Mabsadd L
.60286 .54056 0 2.5 Mabsadd L closepath F
.60381 .53891 0 2.5 Mabsadd m
.60381 .53891 -0.56518 .7779 Mabsadd L
.60381 .53891 -2.37764 .77254 Mabsadd L
.60381 .53891 -0.91448 -0.29713 Mabsadd L
.60381 .53891 -1.46946 -2.02254 Mabsadd L
.60381 .53891 0 -0.96154 Mabsadd L
.60381 .53891 1.46946 -2.02254 Mabsadd L
.60381 .53891 .91448 -0.29713 Mabsadd L
.60381 .53891 2.37764 .77254 Mabsadd L
.60381 .53891 .56518 .7779 Mabsadd L
.60381 .53891 0 2.5 Mabsadd L closepath F
.60476 .53391 0 2.5 Mabsadd m
.60476 .53391 -0.56518 .7779 Mabsadd L
.60476 .53391 -2.37764 .77254 Mabsadd L
.60476 .53391 -0.91448 -0.29713 Mabsadd L
.60476 .53391 -1.46946 -2.02254 Mabsadd L
.60476 .53391 0 -0.96154 Mabsadd L
.60476 .53391 1.46946 -2.02254 Mabsadd L
.60476 .53391 .91448 -0.29713 Mabsadd L
.60476 .53391 2.37764 .77254 Mabsadd L
.60476 .53391 .56518 .7779 Mabsadd L
.60476 .53391 0 2.5 Mabsadd L closepath F
.60571 .51711 0 2.5 Mabsadd m
.60571 .51711 -0.56518 .7779 Mabsadd L
.60571 .51711 -2.37764 .77254 Mabsadd L
.60571 .51711 -0.91448 -0.29713 Mabsadd L
.60571 .51711 -1.46946 -2.02254 Mabsadd L
.60571 .51711 0 -0.96154 Mabsadd L
.60571 .51711 1.46946 -2.02254 Mabsadd L
.60571 .51711 .91448 -0.29713 Mabsadd L
.60571 .51711 2.37764 .77254 Mabsadd L
.60571 .51711 .56518 .7779 Mabsadd L
.60571 .51711 0 2.5 Mabsadd L closepath F
.60667 .51711 0 2.5 Mabsadd m
.60667 .51711 -0.56518 .7779 Mabsadd L
.60667 .51711 -2.37764 .77254 Mabsadd L
.60667 .51711 -0.91448 -0.29713 Mabsadd L
.60667 .51711 -1.46946 -2.02254 Mabsadd L
.60667 .51711 0 -0.96154 Mabsadd L
.60667 .51711 1.46946 -2.02254 Mabsadd L
.60667 .51711 .91448 -0.29713 Mabsadd L
.60667 .51711 2.37764 .77254 Mabsadd L
.60667 .51711 .56518 .7779 Mabsadd L
.60667 .51711 0 2.5 Mabsadd L closepath F
.60762 .51749 0 2.5 Mabsadd m
.60762 .51749 -0.56518 .7779 Mabsadd L
.60762 .51749 -2.37764 .77254 Mabsadd L
.60762 .51749 -0.91448 -0.29713 Mabsadd L
.60762 .51749 -1.46946 -2.02254 Mabsadd L
.60762 .51749 0 -0.96154 Mabsadd L
.60762 .51749 1.46946 -2.02254 Mabsadd L
.60762 .51749 .91448 -0.29713 Mabsadd L
.60762 .51749 2.37764 .77254 Mabsadd L
.60762 .51749 .56518 .7779 Mabsadd L
.60762 .51749 0 2.5 Mabsadd L closepath F
.60857 .52506 0 2.5 Mabsadd m
.60857 .52506 -0.56518 .7779 Mabsadd L
.60857 .52506 -2.37764 .77254 Mabsadd L
.60857 .52506 -0.91448 -0.29713 Mabsadd L
.60857 .52506 -1.46946 -2.02254 Mabsadd L
.60857 .52506 0 -0.96154 Mabsadd L
.60857 .52506 1.46946 -2.02254 Mabsadd L
.60857 .52506 .91448 -0.29713 Mabsadd L
.60857 .52506 2.37764 .77254 Mabsadd L
.60857 .52506 .56518 .7779 Mabsadd L
.60857 .52506 0 2.5 Mabsadd L closepath F
.60952 .53292 0 2.5 Mabsadd m
.60952 .53292 -0.56518 .7779 Mabsadd L
.60952 .53292 -2.37764 .77254 Mabsadd L
.60952 .53292 -0.91448 -0.29713 Mabsadd L
.60952 .53292 -1.46946 -2.02254 Mabsadd L
.60952 .53292 0 -0.96154 Mabsadd L
.60952 .53292 1.46946 -2.02254 Mabsadd L
.60952 .53292 .91448 -0.29713 Mabsadd L
.60952 .53292 2.37764 .77254 Mabsadd L
.60952 .53292 .56518 .7779 Mabsadd L
.60952 .53292 0 2.5 Mabsadd L closepath F
.61048 .53331 0 2.5 Mabsadd m
.61048 .53331 -0.56518 .7779 Mabsadd L
.61048 .53331 -2.37764 .77254 Mabsadd L
.61048 .53331 -0.91448 -0.29713 Mabsadd L
.61048 .53331 -1.46946 -2.02254 Mabsadd L
.61048 .53331 0 -0.96154 Mabsadd L
.61048 .53331 1.46946 -2.02254 Mabsadd L
.61048 .53331 .91448 -0.29713 Mabsadd L
.61048 .53331 2.37764 .77254 Mabsadd L
.61048 .53331 .56518 .7779 Mabsadd L
.61048 .53331 0 2.5 Mabsadd L closepath F
.61143 .5341 0 2.5 Mabsadd m
.61143 .5341 -0.56518 .7779 Mabsadd L
.61143 .5341 -2.37764 .77254 Mabsadd L
.61143 .5341 -0.91448 -0.29713 Mabsadd L
.61143 .5341 -1.46946 -2.02254 Mabsadd L
.61143 .5341 0 -0.96154 Mabsadd L
.61143 .5341 1.46946 -2.02254 Mabsadd L
.61143 .5341 .91448 -0.29713 Mabsadd L
.61143 .5341 2.37764 .77254 Mabsadd L
.61143 .5341 .56518 .7779 Mabsadd L
.61143 .5341 0 2.5 Mabsadd L closepath F
.61238 .53789 0 2.5 Mabsadd m
.61238 .53789 -0.56518 .7779 Mabsadd L
.61238 .53789 -2.37764 .77254 Mabsadd L
.61238 .53789 -0.91448 -0.29713 Mabsadd L
.61238 .53789 -1.46946 -2.02254 Mabsadd L
.61238 .53789 0 -0.96154 Mabsadd L
.61238 .53789 1.46946 -2.02254 Mabsadd L
.61238 .53789 .91448 -0.29713 Mabsadd L
.61238 .53789 2.37764 .77254 Mabsadd L
.61238 .53789 .56518 .7779 Mabsadd L
.61238 .53789 0 2.5 Mabsadd L closepath F
.61333 .53954 0 2.5 Mabsadd m
.61333 .53954 -0.56518 .7779 Mabsadd L
.61333 .53954 -2.37764 .77254 Mabsadd L
.61333 .53954 -0.91448 -0.29713 Mabsadd L
.61333 .53954 -1.46946 -2.02254 Mabsadd L
.61333 .53954 0 -0.96154 Mabsadd L
.61333 .53954 1.46946 -2.02254 Mabsadd L
.61333 .53954 .91448 -0.29713 Mabsadd L
.61333 .53954 2.37764 .77254 Mabsadd L
.61333 .53954 .56518 .7779 Mabsadd L
.61333 .53954 0 2.5 Mabsadd L closepath F
.61429 .53928 0 2.5 Mabsadd m
.61429 .53928 -0.56518 .7779 Mabsadd L
.61429 .53928 -2.37764 .77254 Mabsadd L
.61429 .53928 -0.91448 -0.29713 Mabsadd L
.61429 .53928 -1.46946 -2.02254 Mabsadd L
.61429 .53928 0 -0.96154 Mabsadd L
.61429 .53928 1.46946 -2.02254 Mabsadd L
.61429 .53928 .91448 -0.29713 Mabsadd L
.61429 .53928 2.37764 .77254 Mabsadd L
.61429 .53928 .56518 .7779 Mabsadd L
.61429 .53928 0 2.5 Mabsadd L closepath F
.61524 .58343 0 2.5 Mabsadd m
.61524 .58343 -0.56518 .7779 Mabsadd L
.61524 .58343 -2.37764 .77254 Mabsadd L
.61524 .58343 -0.91448 -0.29713 Mabsadd L
.61524 .58343 -1.46946 -2.02254 Mabsadd L
.61524 .58343 0 -0.96154 Mabsadd L
.61524 .58343 1.46946 -2.02254 Mabsadd L
.61524 .58343 .91448 -0.29713 Mabsadd L
.61524 .58343 2.37764 .77254 Mabsadd L
.61524 .58343 .56518 .7779 Mabsadd L
.61524 .58343 0 2.5 Mabsadd L closepath F
.61619 .58868 0 2.5 Mabsadd m
.61619 .58868 -0.56518 .7779 Mabsadd L
.61619 .58868 -2.37764 .77254 Mabsadd L
.61619 .58868 -0.91448 -0.29713 Mabsadd L
.61619 .58868 -1.46946 -2.02254 Mabsadd L
.61619 .58868 0 -0.96154 Mabsadd L
.61619 .58868 1.46946 -2.02254 Mabsadd L
.61619 .58868 .91448 -0.29713 Mabsadd L
.61619 .58868 2.37764 .77254 Mabsadd L
.61619 .58868 .56518 .7779 Mabsadd L
.61619 .58868 0 2.5 Mabsadd L closepath F
.61714 .60547 0 2.5 Mabsadd m
.61714 .60547 -0.56518 .7779 Mabsadd L
.61714 .60547 -2.37764 .77254 Mabsadd L
.61714 .60547 -0.91448 -0.29713 Mabsadd L
.61714 .60547 -1.46946 -2.02254 Mabsadd L
.61714 .60547 0 -0.96154 Mabsadd L
.61714 .60547 1.46946 -2.02254 Mabsadd L
.61714 .60547 .91448 -0.29713 Mabsadd L
.61714 .60547 2.37764 .77254 Mabsadd L
.61714 .60547 .56518 .7779 Mabsadd L
.61714 .60547 0 2.5 Mabsadd L closepath F
.6181 .6055 0 2.5 Mabsadd m
.6181 .6055 -0.56518 .7779 Mabsadd L
.6181 .6055 -2.37764 .77254 Mabsadd L
.6181 .6055 -0.91448 -0.29713 Mabsadd L
.6181 .6055 -1.46946 -2.02254 Mabsadd L
.6181 .6055 0 -0.96154 Mabsadd L
.6181 .6055 1.46946 -2.02254 Mabsadd L
.6181 .6055 .91448 -0.29713 Mabsadd L
.6181 .6055 2.37764 .77254 Mabsadd L
.6181 .6055 .56518 .7779 Mabsadd L
.6181 .6055 0 2.5 Mabsadd L closepath F
.61905 .60561 0 2.5 Mabsadd m
.61905 .60561 -0.56518 .7779 Mabsadd L
.61905 .60561 -2.37764 .77254 Mabsadd L
.61905 .60561 -0.91448 -0.29713 Mabsadd L
.61905 .60561 -1.46946 -2.02254 Mabsadd L
.61905 .60561 0 -0.96154 Mabsadd L
.61905 .60561 1.46946 -2.02254 Mabsadd L
.61905 .60561 .91448 -0.29713 Mabsadd L
.61905 .60561 2.37764 .77254 Mabsadd L
.61905 .60561 .56518 .7779 Mabsadd L
.61905 .60561 0 2.5 Mabsadd L closepath F
.62 .60565 0 2.5 Mabsadd m
.62 .60565 -0.56518 .7779 Mabsadd L
.62 .60565 -2.37764 .77254 Mabsadd L
.62 .60565 -0.91448 -0.29713 Mabsadd L
.62 .60565 -1.46946 -2.02254 Mabsadd L
.62 .60565 0 -0.96154 Mabsadd L
.62 .60565 1.46946 -2.02254 Mabsadd L
.62 .60565 .91448 -0.29713 Mabsadd L
.62 .60565 2.37764 .77254 Mabsadd L
.62 .60565 .56518 .7779 Mabsadd L
.62 .60565 0 2.5 Mabsadd L closepath F
.62095 .6056 0 2.5 Mabsadd m
.62095 .6056 -0.56518 .7779 Mabsadd L
.62095 .6056 -2.37764 .77254 Mabsadd L
.62095 .6056 -0.91448 -0.29713 Mabsadd L
.62095 .6056 -1.46946 -2.02254 Mabsadd L
.62095 .6056 0 -0.96154 Mabsadd L
.62095 .6056 1.46946 -2.02254 Mabsadd L
.62095 .6056 .91448 -0.29713 Mabsadd L
.62095 .6056 2.37764 .77254 Mabsadd L
.62095 .6056 .56518 .7779 Mabsadd L
.62095 .6056 0 2.5 Mabsadd L closepath F
.6219 .60549 0 2.5 Mabsadd m
.6219 .60549 -0.56518 .7779 Mabsadd L
.6219 .60549 -2.37764 .77254 Mabsadd L
.6219 .60549 -0.91448 -0.29713 Mabsadd L
.6219 .60549 -1.46946 -2.02254 Mabsadd L
.6219 .60549 0 -0.96154 Mabsadd L
.6219 .60549 1.46946 -2.02254 Mabsadd L
.6219 .60549 .91448 -0.29713 Mabsadd L
.6219 .60549 2.37764 .77254 Mabsadd L
.6219 .60549 .56518 .7779 Mabsadd L
.6219 .60549 0 2.5 Mabsadd L closepath F
.62286 .60537 0 2.5 Mabsadd m
.62286 .60537 -0.56518 .7779 Mabsadd L
.62286 .60537 -2.37764 .77254 Mabsadd L
.62286 .60537 -0.91448 -0.29713 Mabsadd L
.62286 .60537 -1.46946 -2.02254 Mabsadd L
.62286 .60537 0 -0.96154 Mabsadd L
.62286 .60537 1.46946 -2.02254 Mabsadd L
.62286 .60537 .91448 -0.29713 Mabsadd L
.62286 .60537 2.37764 .77254 Mabsadd L
.62286 .60537 .56518 .7779 Mabsadd L
.62286 .60537 0 2.5 Mabsadd L closepath F
.62381 .60559 0 2.5 Mabsadd m
.62381 .60559 -0.56518 .7779 Mabsadd L
.62381 .60559 -2.37764 .77254 Mabsadd L
.62381 .60559 -0.91448 -0.29713 Mabsadd L
.62381 .60559 -1.46946 -2.02254 Mabsadd L
.62381 .60559 0 -0.96154 Mabsadd L
.62381 .60559 1.46946 -2.02254 Mabsadd L
.62381 .60559 .91448 -0.29713 Mabsadd L
.62381 .60559 2.37764 .77254 Mabsadd L
.62381 .60559 .56518 .7779 Mabsadd L
.62381 .60559 0 2.5 Mabsadd L closepath F
.62476 .60626 0 2.5 Mabsadd m
.62476 .60626 -0.56518 .7779 Mabsadd L
.62476 .60626 -2.37764 .77254 Mabsadd L
.62476 .60626 -0.91448 -0.29713 Mabsadd L
.62476 .60626 -1.46946 -2.02254 Mabsadd L
.62476 .60626 0 -0.96154 Mabsadd L
.62476 .60626 1.46946 -2.02254 Mabsadd L
.62476 .60626 .91448 -0.29713 Mabsadd L
.62476 .60626 2.37764 .77254 Mabsadd L
.62476 .60626 .56518 .7779 Mabsadd L
.62476 .60626 0 2.5 Mabsadd L closepath F
.62571 .60631 0 2.5 Mabsadd m
.62571 .60631 -0.56518 .7779 Mabsadd L
.62571 .60631 -2.37764 .77254 Mabsadd L
.62571 .60631 -0.91448 -0.29713 Mabsadd L
.62571 .60631 -1.46946 -2.02254 Mabsadd L
.62571 .60631 0 -0.96154 Mabsadd L
.62571 .60631 1.46946 -2.02254 Mabsadd L
.62571 .60631 .91448 -0.29713 Mabsadd L
.62571 .60631 2.37764 .77254 Mabsadd L
.62571 .60631 .56518 .7779 Mabsadd L
.62571 .60631 0 2.5 Mabsadd L closepath F
.62667 .60645 0 2.5 Mabsadd m
.62667 .60645 -0.56518 .7779 Mabsadd L
.62667 .60645 -2.37764 .77254 Mabsadd L
.62667 .60645 -0.91448 -0.29713 Mabsadd L
.62667 .60645 -1.46946 -2.02254 Mabsadd L
.62667 .60645 0 -0.96154 Mabsadd L
.62667 .60645 1.46946 -2.02254 Mabsadd L
.62667 .60645 .91448 -0.29713 Mabsadd L
.62667 .60645 2.37764 .77254 Mabsadd L
.62667 .60645 .56518 .7779 Mabsadd L
.62667 .60645 0 2.5 Mabsadd L closepath F
.62762 .60646 0 2.5 Mabsadd m
.62762 .60646 -0.56518 .7779 Mabsadd L
.62762 .60646 -2.37764 .77254 Mabsadd L
.62762 .60646 -0.91448 -0.29713 Mabsadd L
.62762 .60646 -1.46946 -2.02254 Mabsadd L
.62762 .60646 0 -0.96154 Mabsadd L
.62762 .60646 1.46946 -2.02254 Mabsadd L
.62762 .60646 .91448 -0.29713 Mabsadd L
.62762 .60646 2.37764 .77254 Mabsadd L
.62762 .60646 .56518 .7779 Mabsadd L
.62762 .60646 0 2.5 Mabsadd L closepath F
.62857 .6065 0 2.5 Mabsadd m
.62857 .6065 -0.56518 .7779 Mabsadd L
.62857 .6065 -2.37764 .77254 Mabsadd L
.62857 .6065 -0.91448 -0.29713 Mabsadd L
.62857 .6065 -1.46946 -2.02254 Mabsadd L
.62857 .6065 0 -0.96154 Mabsadd L
.62857 .6065 1.46946 -2.02254 Mabsadd L
.62857 .6065 .91448 -0.29713 Mabsadd L
.62857 .6065 2.37764 .77254 Mabsadd L
.62857 .6065 .56518 .7779 Mabsadd L
.62857 .6065 0 2.5 Mabsadd L closepath F
.62952 .60651 0 2.5 Mabsadd m
.62952 .60651 -0.56518 .7779 Mabsadd L
.62952 .60651 -2.37764 .77254 Mabsadd L
.62952 .60651 -0.91448 -0.29713 Mabsadd L
.62952 .60651 -1.46946 -2.02254 Mabsadd L
.62952 .60651 0 -0.96154 Mabsadd L
.62952 .60651 1.46946 -2.02254 Mabsadd L
.62952 .60651 .91448 -0.29713 Mabsadd L
.62952 .60651 2.37764 .77254 Mabsadd L
.62952 .60651 .56518 .7779 Mabsadd L
.62952 .60651 0 2.5 Mabsadd L closepath F
.63048 .60652 0 2.5 Mabsadd m
.63048 .60652 -0.56518 .7779 Mabsadd L
.63048 .60652 -2.37764 .77254 Mabsadd L
.63048 .60652 -0.91448 -0.29713 Mabsadd L
.63048 .60652 -1.46946 -2.02254 Mabsadd L
.63048 .60652 0 -0.96154 Mabsadd L
.63048 .60652 1.46946 -2.02254 Mabsadd L
.63048 .60652 .91448 -0.29713 Mabsadd L
.63048 .60652 2.37764 .77254 Mabsadd L
.63048 .60652 .56518 .7779 Mabsadd L
.63048 .60652 0 2.5 Mabsadd L closepath F
.63143 .60654 0 2.5 Mabsadd m
.63143 .60654 -0.56518 .7779 Mabsadd L
.63143 .60654 -2.37764 .77254 Mabsadd L
.63143 .60654 -0.91448 -0.29713 Mabsadd L
.63143 .60654 -1.46946 -2.02254 Mabsadd L
.63143 .60654 0 -0.96154 Mabsadd L
.63143 .60654 1.46946 -2.02254 Mabsadd L
.63143 .60654 .91448 -0.29713 Mabsadd L
.63143 .60654 2.37764 .77254 Mabsadd L
.63143 .60654 .56518 .7779 Mabsadd L
.63143 .60654 0 2.5 Mabsadd L closepath F
.63238 .60654 0 2.5 Mabsadd m
.63238 .60654 -0.56518 .7779 Mabsadd L
.63238 .60654 -2.37764 .77254 Mabsadd L
.63238 .60654 -0.91448 -0.29713 Mabsadd L
.63238 .60654 -1.46946 -2.02254 Mabsadd L
.63238 .60654 0 -0.96154 Mabsadd L
.63238 .60654 1.46946 -2.02254 Mabsadd L
.63238 .60654 .91448 -0.29713 Mabsadd L
.63238 .60654 2.37764 .77254 Mabsadd L
.63238 .60654 .56518 .7779 Mabsadd L
.63238 .60654 0 2.5 Mabsadd L closepath F
.63333 .60654 0 2.5 Mabsadd m
.63333 .60654 -0.56518 .7779 Mabsadd L
.63333 .60654 -2.37764 .77254 Mabsadd L
.63333 .60654 -0.91448 -0.29713 Mabsadd L
.63333 .60654 -1.46946 -2.02254 Mabsadd L
.63333 .60654 0 -0.96154 Mabsadd L
.63333 .60654 1.46946 -2.02254 Mabsadd L
.63333 .60654 .91448 -0.29713 Mabsadd L
.63333 .60654 2.37764 .77254 Mabsadd L
.63333 .60654 .56518 .7779 Mabsadd L
.63333 .60654 0 2.5 Mabsadd L closepath F
.63429 .60655 0 2.5 Mabsadd m
.63429 .60655 -0.56518 .7779 Mabsadd L
.63429 .60655 -2.37764 .77254 Mabsadd L
.63429 .60655 -0.91448 -0.29713 Mabsadd L
.63429 .60655 -1.46946 -2.02254 Mabsadd L
.63429 .60655 0 -0.96154 Mabsadd L
.63429 .60655 1.46946 -2.02254 Mabsadd L
.63429 .60655 .91448 -0.29713 Mabsadd L
.63429 .60655 2.37764 .77254 Mabsadd L
.63429 .60655 .56518 .7779 Mabsadd L
.63429 .60655 0 2.5 Mabsadd L closepath F
.63524 .60656 0 2.5 Mabsadd m
.63524 .60656 -0.56518 .7779 Mabsadd L
.63524 .60656 -2.37764 .77254 Mabsadd L
.63524 .60656 -0.91448 -0.29713 Mabsadd L
.63524 .60656 -1.46946 -2.02254 Mabsadd L
.63524 .60656 0 -0.96154 Mabsadd L
.63524 .60656 1.46946 -2.02254 Mabsadd L
.63524 .60656 .91448 -0.29713 Mabsadd L
.63524 .60656 2.37764 .77254 Mabsadd L
.63524 .60656 .56518 .7779 Mabsadd L
.63524 .60656 0 2.5 Mabsadd L closepath F
.63619 .60656 0 2.5 Mabsadd m
.63619 .60656 -0.56518 .7779 Mabsadd L
.63619 .60656 -2.37764 .77254 Mabsadd L
.63619 .60656 -0.91448 -0.29713 Mabsadd L
.63619 .60656 -1.46946 -2.02254 Mabsadd L
.63619 .60656 0 -0.96154 Mabsadd L
.63619 .60656 1.46946 -2.02254 Mabsadd L
.63619 .60656 .91448 -0.29713 Mabsadd L
.63619 .60656 2.37764 .77254 Mabsadd L
.63619 .60656 .56518 .7779 Mabsadd L
.63619 .60656 0 2.5 Mabsadd L closepath F
.63714 .60656 0 2.5 Mabsadd m
.63714 .60656 -0.56518 .7779 Mabsadd L
.63714 .60656 -2.37764 .77254 Mabsadd L
.63714 .60656 -0.91448 -0.29713 Mabsadd L
.63714 .60656 -1.46946 -2.02254 Mabsadd L
.63714 .60656 0 -0.96154 Mabsadd L
.63714 .60656 1.46946 -2.02254 Mabsadd L
.63714 .60656 .91448 -0.29713 Mabsadd L
.63714 .60656 2.37764 .77254 Mabsadd L
.63714 .60656 .56518 .7779 Mabsadd L
.63714 .60656 0 2.5 Mabsadd L closepath F
.6381 .60656 0 2.5 Mabsadd m
.6381 .60656 -0.56518 .7779 Mabsadd L
.6381 .60656 -2.37764 .77254 Mabsadd L
.6381 .60656 -0.91448 -0.29713 Mabsadd L
.6381 .60656 -1.46946 -2.02254 Mabsadd L
.6381 .60656 0 -0.96154 Mabsadd L
.6381 .60656 1.46946 -2.02254 Mabsadd L
.6381 .60656 .91448 -0.29713 Mabsadd L
.6381 .60656 2.37764 .77254 Mabsadd L
.6381 .60656 .56518 .7779 Mabsadd L
.6381 .60656 0 2.5 Mabsadd L closepath F
.63905 .60656 0 2.5 Mabsadd m
.63905 .60656 -0.56518 .7779 Mabsadd L
.63905 .60656 -2.37764 .77254 Mabsadd L
.63905 .60656 -0.91448 -0.29713 Mabsadd L
.63905 .60656 -1.46946 -2.02254 Mabsadd L
.63905 .60656 0 -0.96154 Mabsadd L
.63905 .60656 1.46946 -2.02254 Mabsadd L
.63905 .60656 .91448 -0.29713 Mabsadd L
.63905 .60656 2.37764 .77254 Mabsadd L
.63905 .60656 .56518 .7779 Mabsadd L
.63905 .60656 0 2.5 Mabsadd L closepath F
.64 .60656 0 2.5 Mabsadd m
.64 .60656 -0.56518 .7779 Mabsadd L
.64 .60656 -2.37764 .77254 Mabsadd L
.64 .60656 -0.91448 -0.29713 Mabsadd L
.64 .60656 -1.46946 -2.02254 Mabsadd L
.64 .60656 0 -0.96154 Mabsadd L
.64 .60656 1.46946 -2.02254 Mabsadd L
.64 .60656 .91448 -0.29713 Mabsadd L
.64 .60656 2.37764 .77254 Mabsadd L
.64 .60656 .56518 .7779 Mabsadd L
.64 .60656 0 2.5 Mabsadd L closepath F
.64095 .60656 0 2.5 Mabsadd m
.64095 .60656 -0.56518 .7779 Mabsadd L
.64095 .60656 -2.37764 .77254 Mabsadd L
.64095 .60656 -0.91448 -0.29713 Mabsadd L
.64095 .60656 -1.46946 -2.02254 Mabsadd L
.64095 .60656 0 -0.96154 Mabsadd L
.64095 .60656 1.46946 -2.02254 Mabsadd L
.64095 .60656 .91448 -0.29713 Mabsadd L
.64095 .60656 2.37764 .77254 Mabsadd L
.64095 .60656 .56518 .7779 Mabsadd L
.64095 .60656 0 2.5 Mabsadd L closepath F
.6419 .60655 0 2.5 Mabsadd m
.6419 .60655 -0.56518 .7779 Mabsadd L
.6419 .60655 -2.37764 .77254 Mabsadd L
.6419 .60655 -0.91448 -0.29713 Mabsadd L
.6419 .60655 -1.46946 -2.02254 Mabsadd L
.6419 .60655 0 -0.96154 Mabsadd L
.6419 .60655 1.46946 -2.02254 Mabsadd L
.6419 .60655 .91448 -0.29713 Mabsadd L
.6419 .60655 2.37764 .77254 Mabsadd L
.6419 .60655 .56518 .7779 Mabsadd L
.6419 .60655 0 2.5 Mabsadd L closepath F
.64286 .60655 0 2.5 Mabsadd m
.64286 .60655 -0.56518 .7779 Mabsadd L
.64286 .60655 -2.37764 .77254 Mabsadd L
.64286 .60655 -0.91448 -0.29713 Mabsadd L
.64286 .60655 -1.46946 -2.02254 Mabsadd L
.64286 .60655 0 -0.96154 Mabsadd L
.64286 .60655 1.46946 -2.02254 Mabsadd L
.64286 .60655 .91448 -0.29713 Mabsadd L
.64286 .60655 2.37764 .77254 Mabsadd L
.64286 .60655 .56518 .7779 Mabsadd L
.64286 .60655 0 2.5 Mabsadd L closepath F
.64381 .60655 0 2.5 Mabsadd m
.64381 .60655 -0.56518 .7779 Mabsadd L
.64381 .60655 -2.37764 .77254 Mabsadd L
.64381 .60655 -0.91448 -0.29713 Mabsadd L
.64381 .60655 -1.46946 -2.02254 Mabsadd L
.64381 .60655 0 -0.96154 Mabsadd L
.64381 .60655 1.46946 -2.02254 Mabsadd L
.64381 .60655 .91448 -0.29713 Mabsadd L
.64381 .60655 2.37764 .77254 Mabsadd L
.64381 .60655 .56518 .7779 Mabsadd L
.64381 .60655 0 2.5 Mabsadd L closepath F
.64476 .60655 0 2.5 Mabsadd m
.64476 .60655 -0.56518 .7779 Mabsadd L
.64476 .60655 -2.37764 .77254 Mabsadd L
.64476 .60655 -0.91448 -0.29713 Mabsadd L
.64476 .60655 -1.46946 -2.02254 Mabsadd L
.64476 .60655 0 -0.96154 Mabsadd L
.64476 .60655 1.46946 -2.02254 Mabsadd L
.64476 .60655 .91448 -0.29713 Mabsadd L
.64476 .60655 2.37764 .77254 Mabsadd L
.64476 .60655 .56518 .7779 Mabsadd L
.64476 .60655 0 2.5 Mabsadd L closepath F
.64571 .60654 0 2.5 Mabsadd m
.64571 .60654 -0.56518 .7779 Mabsadd L
.64571 .60654 -2.37764 .77254 Mabsadd L
.64571 .60654 -0.91448 -0.29713 Mabsadd L
.64571 .60654 -1.46946 -2.02254 Mabsadd L
.64571 .60654 0 -0.96154 Mabsadd L
.64571 .60654 1.46946 -2.02254 Mabsadd L
.64571 .60654 .91448 -0.29713 Mabsadd L
.64571 .60654 2.37764 .77254 Mabsadd L
.64571 .60654 .56518 .7779 Mabsadd L
.64571 .60654 0 2.5 Mabsadd L closepath F
.64667 .60654 0 2.5 Mabsadd m
.64667 .60654 -0.56518 .7779 Mabsadd L
.64667 .60654 -2.37764 .77254 Mabsadd L
.64667 .60654 -0.91448 -0.29713 Mabsadd L
.64667 .60654 -1.46946 -2.02254 Mabsadd L
.64667 .60654 0 -0.96154 Mabsadd L
.64667 .60654 1.46946 -2.02254 Mabsadd L
.64667 .60654 .91448 -0.29713 Mabsadd L
.64667 .60654 2.37764 .77254 Mabsadd L
.64667 .60654 .56518 .7779 Mabsadd L
.64667 .60654 0 2.5 Mabsadd L closepath F
.64762 .60653 0 2.5 Mabsadd m
.64762 .60653 -0.56518 .7779 Mabsadd L
.64762 .60653 -2.37764 .77254 Mabsadd L
.64762 .60653 -0.91448 -0.29713 Mabsadd L
.64762 .60653 -1.46946 -2.02254 Mabsadd L
.64762 .60653 0 -0.96154 Mabsadd L
.64762 .60653 1.46946 -2.02254 Mabsadd L
.64762 .60653 .91448 -0.29713 Mabsadd L
.64762 .60653 2.37764 .77254 Mabsadd L
.64762 .60653 .56518 .7779 Mabsadd L
.64762 .60653 0 2.5 Mabsadd L closepath F
.64857 .60651 0 2.5 Mabsadd m
.64857 .60651 -0.56518 .7779 Mabsadd L
.64857 .60651 -2.37764 .77254 Mabsadd L
.64857 .60651 -0.91448 -0.29713 Mabsadd L
.64857 .60651 -1.46946 -2.02254 Mabsadd L
.64857 .60651 0 -0.96154 Mabsadd L
.64857 .60651 1.46946 -2.02254 Mabsadd L
.64857 .60651 .91448 -0.29713 Mabsadd L
.64857 .60651 2.37764 .77254 Mabsadd L
.64857 .60651 .56518 .7779 Mabsadd L
.64857 .60651 0 2.5 Mabsadd L closepath F
.64952 .60651 0 2.5 Mabsadd m
.64952 .60651 -0.56518 .7779 Mabsadd L
.64952 .60651 -2.37764 .77254 Mabsadd L
.64952 .60651 -0.91448 -0.29713 Mabsadd L
.64952 .60651 -1.46946 -2.02254 Mabsadd L
.64952 .60651 0 -0.96154 Mabsadd L
.64952 .60651 1.46946 -2.02254 Mabsadd L
.64952 .60651 .91448 -0.29713 Mabsadd L
.64952 .60651 2.37764 .77254 Mabsadd L
.64952 .60651 .56518 .7779 Mabsadd L
.64952 .60651 0 2.5 Mabsadd L closepath F
.65048 .60648 0 2.5 Mabsadd m
.65048 .60648 -0.56518 .7779 Mabsadd L
.65048 .60648 -2.37764 .77254 Mabsadd L
.65048 .60648 -0.91448 -0.29713 Mabsadd L
.65048 .60648 -1.46946 -2.02254 Mabsadd L
.65048 .60648 0 -0.96154 Mabsadd L
.65048 .60648 1.46946 -2.02254 Mabsadd L
.65048 .60648 .91448 -0.29713 Mabsadd L
.65048 .60648 2.37764 .77254 Mabsadd L
.65048 .60648 .56518 .7779 Mabsadd L
.65048 .60648 0 2.5 Mabsadd L closepath F
.65143 .60638 0 2.5 Mabsadd m
.65143 .60638 -0.56518 .7779 Mabsadd L
.65143 .60638 -2.37764 .77254 Mabsadd L
.65143 .60638 -0.91448 -0.29713 Mabsadd L
.65143 .60638 -1.46946 -2.02254 Mabsadd L
.65143 .60638 0 -0.96154 Mabsadd L
.65143 .60638 1.46946 -2.02254 Mabsadd L
.65143 .60638 .91448 -0.29713 Mabsadd L
.65143 .60638 2.37764 .77254 Mabsadd L
.65143 .60638 .56518 .7779 Mabsadd L
.65143 .60638 0 2.5 Mabsadd L closepath F
.65238 .60636 0 2.5 Mabsadd m
.65238 .60636 -0.56518 .7779 Mabsadd L
.65238 .60636 -2.37764 .77254 Mabsadd L
.65238 .60636 -0.91448 -0.29713 Mabsadd L
.65238 .60636 -1.46946 -2.02254 Mabsadd L
.65238 .60636 0 -0.96154 Mabsadd L
.65238 .60636 1.46946 -2.02254 Mabsadd L
.65238 .60636 .91448 -0.29713 Mabsadd L
.65238 .60636 2.37764 .77254 Mabsadd L
.65238 .60636 .56518 .7779 Mabsadd L
.65238 .60636 0 2.5 Mabsadd L closepath F
.65333 .60618 0 2.5 Mabsadd m
.65333 .60618 -0.56518 .7779 Mabsadd L
.65333 .60618 -2.37764 .77254 Mabsadd L
.65333 .60618 -0.91448 -0.29713 Mabsadd L
.65333 .60618 -1.46946 -2.02254 Mabsadd L
.65333 .60618 0 -0.96154 Mabsadd L
.65333 .60618 1.46946 -2.02254 Mabsadd L
.65333 .60618 .91448 -0.29713 Mabsadd L
.65333 .60618 2.37764 .77254 Mabsadd L
.65333 .60618 .56518 .7779 Mabsadd L
.65333 .60618 0 2.5 Mabsadd L closepath F
.65429 .60609 0 2.5 Mabsadd m
.65429 .60609 -0.56518 .7779 Mabsadd L
.65429 .60609 -2.37764 .77254 Mabsadd L
.65429 .60609 -0.91448 -0.29713 Mabsadd L
.65429 .60609 -1.46946 -2.02254 Mabsadd L
.65429 .60609 0 -0.96154 Mabsadd L
.65429 .60609 1.46946 -2.02254 Mabsadd L
.65429 .60609 .91448 -0.29713 Mabsadd L
.65429 .60609 2.37764 .77254 Mabsadd L
.65429 .60609 .56518 .7779 Mabsadd L
.65429 .60609 0 2.5 Mabsadd L closepath F
.65524 .60518 0 2.5 Mabsadd m
.65524 .60518 -0.56518 .7779 Mabsadd L
.65524 .60518 -2.37764 .77254 Mabsadd L
.65524 .60518 -0.91448 -0.29713 Mabsadd L
.65524 .60518 -1.46946 -2.02254 Mabsadd L
.65524 .60518 0 -0.96154 Mabsadd L
.65524 .60518 1.46946 -2.02254 Mabsadd L
.65524 .60518 .91448 -0.29713 Mabsadd L
.65524 .60518 2.37764 .77254 Mabsadd L
.65524 .60518 .56518 .7779 Mabsadd L
.65524 .60518 0 2.5 Mabsadd L closepath F
.65619 .60466 0 2.5 Mabsadd m
.65619 .60466 -0.56518 .7779 Mabsadd L
.65619 .60466 -2.37764 .77254 Mabsadd L
.65619 .60466 -0.91448 -0.29713 Mabsadd L
.65619 .60466 -1.46946 -2.02254 Mabsadd L
.65619 .60466 0 -0.96154 Mabsadd L
.65619 .60466 1.46946 -2.02254 Mabsadd L
.65619 .60466 .91448 -0.29713 Mabsadd L
.65619 .60466 2.37764 .77254 Mabsadd L
.65619 .60466 .56518 .7779 Mabsadd L
.65619 .60466 0 2.5 Mabsadd L closepath F
.65714 .60322 0 2.5 Mabsadd m
.65714 .60322 -0.56518 .7779 Mabsadd L
.65714 .60322 -2.37764 .77254 Mabsadd L
.65714 .60322 -0.91448 -0.29713 Mabsadd L
.65714 .60322 -1.46946 -2.02254 Mabsadd L
.65714 .60322 0 -0.96154 Mabsadd L
.65714 .60322 1.46946 -2.02254 Mabsadd L
.65714 .60322 .91448 -0.29713 Mabsadd L
.65714 .60322 2.37764 .77254 Mabsadd L
.65714 .60322 .56518 .7779 Mabsadd L
.65714 .60322 0 2.5 Mabsadd L closepath F
.6581 .59891 0 2.5 Mabsadd m
.6581 .59891 -0.56518 .7779 Mabsadd L
.6581 .59891 -2.37764 .77254 Mabsadd L
.6581 .59891 -0.91448 -0.29713 Mabsadd L
.6581 .59891 -1.46946 -2.02254 Mabsadd L
.6581 .59891 0 -0.96154 Mabsadd L
.6581 .59891 1.46946 -2.02254 Mabsadd L
.6581 .59891 .91448 -0.29713 Mabsadd L
.6581 .59891 2.37764 .77254 Mabsadd L
.6581 .59891 .56518 .7779 Mabsadd L
.6581 .59891 0 2.5 Mabsadd L closepath F
.65905 .58464 0 2.5 Mabsadd m
.65905 .58464 -0.56518 .7779 Mabsadd L
.65905 .58464 -2.37764 .77254 Mabsadd L
.65905 .58464 -0.91448 -0.29713 Mabsadd L
.65905 .58464 -1.46946 -2.02254 Mabsadd L
.65905 .58464 0 -0.96154 Mabsadd L
.65905 .58464 1.46946 -2.02254 Mabsadd L
.65905 .58464 .91448 -0.29713 Mabsadd L
.65905 .58464 2.37764 .77254 Mabsadd L
.65905 .58464 .56518 .7779 Mabsadd L
.65905 .58464 0 2.5 Mabsadd L closepath F
.66 .58464 0 2.5 Mabsadd m
.66 .58464 -0.56518 .7779 Mabsadd L
.66 .58464 -2.37764 .77254 Mabsadd L
.66 .58464 -0.91448 -0.29713 Mabsadd L
.66 .58464 -1.46946 -2.02254 Mabsadd L
.66 .58464 0 -0.96154 Mabsadd L
.66 .58464 1.46946 -2.02254 Mabsadd L
.66 .58464 .91448 -0.29713 Mabsadd L
.66 .58464 2.37764 .77254 Mabsadd L
.66 .58464 .56518 .7779 Mabsadd L
.66 .58464 0 2.5 Mabsadd L closepath F
.66095 .58492 0 2.5 Mabsadd m
.66095 .58492 -0.56518 .7779 Mabsadd L
.66095 .58492 -2.37764 .77254 Mabsadd L
.66095 .58492 -0.91448 -0.29713 Mabsadd L
.66095 .58492 -1.46946 -2.02254 Mabsadd L
.66095 .58492 0 -0.96154 Mabsadd L
.66095 .58492 1.46946 -2.02254 Mabsadd L
.66095 .58492 .91448 -0.29713 Mabsadd L
.66095 .58492 2.37764 .77254 Mabsadd L
.66095 .58492 .56518 .7779 Mabsadd L
.66095 .58492 0 2.5 Mabsadd L closepath F
.6619 .59059 0 2.5 Mabsadd m
.6619 .59059 -0.56518 .7779 Mabsadd L
.6619 .59059 -2.37764 .77254 Mabsadd L
.6619 .59059 -0.91448 -0.29713 Mabsadd L
.6619 .59059 -1.46946 -2.02254 Mabsadd L
.6619 .59059 0 -0.96154 Mabsadd L
.6619 .59059 1.46946 -2.02254 Mabsadd L
.6619 .59059 .91448 -0.29713 Mabsadd L
.6619 .59059 2.37764 .77254 Mabsadd L
.6619 .59059 .56518 .7779 Mabsadd L
.6619 .59059 0 2.5 Mabsadd L closepath F
.66286 .59618 0 2.5 Mabsadd m
.66286 .59618 -0.56518 .7779 Mabsadd L
.66286 .59618 -2.37764 .77254 Mabsadd L
.66286 .59618 -0.91448 -0.29713 Mabsadd L
.66286 .59618 -1.46946 -2.02254 Mabsadd L
.66286 .59618 0 -0.96154 Mabsadd L
.66286 .59618 1.46946 -2.02254 Mabsadd L
.66286 .59618 .91448 -0.29713 Mabsadd L
.66286 .59618 2.37764 .77254 Mabsadd L
.66286 .59618 .56518 .7779 Mabsadd L
.66286 .59618 0 2.5 Mabsadd L closepath F
.66381 .59642 0 2.5 Mabsadd m
.66381 .59642 -0.56518 .7779 Mabsadd L
.66381 .59642 -2.37764 .77254 Mabsadd L
.66381 .59642 -0.91448 -0.29713 Mabsadd L
.66381 .59642 -1.46946 -2.02254 Mabsadd L
.66381 .59642 0 -0.96154 Mabsadd L
.66381 .59642 1.46946 -2.02254 Mabsadd L
.66381 .59642 .91448 -0.29713 Mabsadd L
.66381 .59642 2.37764 .77254 Mabsadd L
.66381 .59642 .56518 .7779 Mabsadd L
.66381 .59642 0 2.5 Mabsadd L closepath F
.66476 .59699 0 2.5 Mabsadd m
.66476 .59699 -0.56518 .7779 Mabsadd L
.66476 .59699 -2.37764 .77254 Mabsadd L
.66476 .59699 -0.91448 -0.29713 Mabsadd L
.66476 .59699 -1.46946 -2.02254 Mabsadd L
.66476 .59699 0 -0.96154 Mabsadd L
.66476 .59699 1.46946 -2.02254 Mabsadd L
.66476 .59699 .91448 -0.29713 Mabsadd L
.66476 .59699 2.37764 .77254 Mabsadd L
.66476 .59699 .56518 .7779 Mabsadd L
.66476 .59699 0 2.5 Mabsadd L closepath F
.66571 .60043 0 2.5 Mabsadd m
.66571 .60043 -0.56518 .7779 Mabsadd L
.66571 .60043 -2.37764 .77254 Mabsadd L
.66571 .60043 -0.91448 -0.29713 Mabsadd L
.66571 .60043 -1.46946 -2.02254 Mabsadd L
.66571 .60043 0 -0.96154 Mabsadd L
.66571 .60043 1.46946 -2.02254 Mabsadd L
.66571 .60043 .91448 -0.29713 Mabsadd L
.66571 .60043 2.37764 .77254 Mabsadd L
.66571 .60043 .56518 .7779 Mabsadd L
.66571 .60043 0 2.5 Mabsadd L closepath F
.66667 .60268 0 2.5 Mabsadd m
.66667 .60268 -0.56518 .7779 Mabsadd L
.66667 .60268 -2.37764 .77254 Mabsadd L
.66667 .60268 -0.91448 -0.29713 Mabsadd L
.66667 .60268 -1.46946 -2.02254 Mabsadd L
.66667 .60268 0 -0.96154 Mabsadd L
.66667 .60268 1.46946 -2.02254 Mabsadd L
.66667 .60268 .91448 -0.29713 Mabsadd L
.66667 .60268 2.37764 .77254 Mabsadd L
.66667 .60268 .56518 .7779 Mabsadd L
.66667 .60268 0 2.5 Mabsadd L closepath F
.66762 .60242 0 2.5 Mabsadd m
.66762 .60242 -0.56518 .7779 Mabsadd L
.66762 .60242 -2.37764 .77254 Mabsadd L
.66762 .60242 -0.91448 -0.29713 Mabsadd L
.66762 .60242 -1.46946 -2.02254 Mabsadd L
.66762 .60242 0 -0.96154 Mabsadd L
.66762 .60242 1.46946 -2.02254 Mabsadd L
.66762 .60242 .91448 -0.29713 Mabsadd L
.66762 .60242 2.37764 .77254 Mabsadd L
.66762 .60242 .56518 .7779 Mabsadd L
.66762 .60242 0 2.5 Mabsadd L closepath F
.66857 .65566 0 2.5 Mabsadd m
.66857 .65566 -0.56518 .7779 Mabsadd L
.66857 .65566 -2.37764 .77254 Mabsadd L
.66857 .65566 -0.91448 -0.29713 Mabsadd L
.66857 .65566 -1.46946 -2.02254 Mabsadd L
.66857 .65566 0 -0.96154 Mabsadd L
.66857 .65566 1.46946 -2.02254 Mabsadd L
.66857 .65566 .91448 -0.29713 Mabsadd L
.66857 .65566 2.37764 .77254 Mabsadd L
.66857 .65566 .56518 .7779 Mabsadd L
.66857 .65566 0 2.5 Mabsadd L closepath F
.66952 .66282 0 2.5 Mabsadd m
.66952 .66282 -0.56518 .7779 Mabsadd L
.66952 .66282 -2.37764 .77254 Mabsadd L
.66952 .66282 -0.91448 -0.29713 Mabsadd L
.66952 .66282 -1.46946 -2.02254 Mabsadd L
.66952 .66282 0 -0.96154 Mabsadd L
.66952 .66282 1.46946 -2.02254 Mabsadd L
.66952 .66282 .91448 -0.29713 Mabsadd L
.66952 .66282 2.37764 .77254 Mabsadd L
.66952 .66282 .56518 .7779 Mabsadd L
.66952 .66282 0 2.5 Mabsadd L closepath F
.67048 .68653 0 2.5 Mabsadd m
.67048 .68653 -0.56518 .7779 Mabsadd L
.67048 .68653 -2.37764 .77254 Mabsadd L
.67048 .68653 -0.91448 -0.29713 Mabsadd L
.67048 .68653 -1.46946 -2.02254 Mabsadd L
.67048 .68653 0 -0.96154 Mabsadd L
.67048 .68653 1.46946 -2.02254 Mabsadd L
.67048 .68653 .91448 -0.29713 Mabsadd L
.67048 .68653 2.37764 .77254 Mabsadd L
.67048 .68653 .56518 .7779 Mabsadd L
.67048 .68653 0 2.5 Mabsadd L closepath F
.67143 .68658 0 2.5 Mabsadd m
.67143 .68658 -0.56518 .7779 Mabsadd L
.67143 .68658 -2.37764 .77254 Mabsadd L
.67143 .68658 -0.91448 -0.29713 Mabsadd L
.67143 .68658 -1.46946 -2.02254 Mabsadd L
.67143 .68658 0 -0.96154 Mabsadd L
.67143 .68658 1.46946 -2.02254 Mabsadd L
.67143 .68658 .91448 -0.29713 Mabsadd L
.67143 .68658 2.37764 .77254 Mabsadd L
.67143 .68658 .56518 .7779 Mabsadd L
.67143 .68658 0 2.5 Mabsadd L closepath F
.67238 .68672 0 2.5 Mabsadd m
.67238 .68672 -0.56518 .7779 Mabsadd L
.67238 .68672 -2.37764 .77254 Mabsadd L
.67238 .68672 -0.91448 -0.29713 Mabsadd L
.67238 .68672 -1.46946 -2.02254 Mabsadd L
.67238 .68672 0 -0.96154 Mabsadd L
.67238 .68672 1.46946 -2.02254 Mabsadd L
.67238 .68672 .91448 -0.29713 Mabsadd L
.67238 .68672 2.37764 .77254 Mabsadd L
.67238 .68672 .56518 .7779 Mabsadd L
.67238 .68672 0 2.5 Mabsadd L closepath F
.67333 .68678 0 2.5 Mabsadd m
.67333 .68678 -0.56518 .7779 Mabsadd L
.67333 .68678 -2.37764 .77254 Mabsadd L
.67333 .68678 -0.91448 -0.29713 Mabsadd L
.67333 .68678 -1.46946 -2.02254 Mabsadd L
.67333 .68678 0 -0.96154 Mabsadd L
.67333 .68678 1.46946 -2.02254 Mabsadd L
.67333 .68678 .91448 -0.29713 Mabsadd L
.67333 .68678 2.37764 .77254 Mabsadd L
.67333 .68678 .56518 .7779 Mabsadd L
.67333 .68678 0 2.5 Mabsadd L closepath F
.67429 .68672 0 2.5 Mabsadd m
.67429 .68672 -0.56518 .7779 Mabsadd L
.67429 .68672 -2.37764 .77254 Mabsadd L
.67429 .68672 -0.91448 -0.29713 Mabsadd L
.67429 .68672 -1.46946 -2.02254 Mabsadd L
.67429 .68672 0 -0.96154 Mabsadd L
.67429 .68672 1.46946 -2.02254 Mabsadd L
.67429 .68672 .91448 -0.29713 Mabsadd L
.67429 .68672 2.37764 .77254 Mabsadd L
.67429 .68672 .56518 .7779 Mabsadd L
.67429 .68672 0 2.5 Mabsadd L closepath F
.67524 .68659 0 2.5 Mabsadd m
.67524 .68659 -0.56518 .7779 Mabsadd L
.67524 .68659 -2.37764 .77254 Mabsadd L
.67524 .68659 -0.91448 -0.29713 Mabsadd L
.67524 .68659 -1.46946 -2.02254 Mabsadd L
.67524 .68659 0 -0.96154 Mabsadd L
.67524 .68659 1.46946 -2.02254 Mabsadd L
.67524 .68659 .91448 -0.29713 Mabsadd L
.67524 .68659 2.37764 .77254 Mabsadd L
.67524 .68659 .56518 .7779 Mabsadd L
.67524 .68659 0 2.5 Mabsadd L closepath F
.67619 .68717 0 2.5 Mabsadd m
.67619 .68717 -0.56518 .7779 Mabsadd L
.67619 .68717 -2.37764 .77254 Mabsadd L
.67619 .68717 -0.91448 -0.29713 Mabsadd L
.67619 .68717 -1.46946 -2.02254 Mabsadd L
.67619 .68717 0 -0.96154 Mabsadd L
.67619 .68717 1.46946 -2.02254 Mabsadd L
.67619 .68717 .91448 -0.29713 Mabsadd L
.67619 .68717 2.37764 .77254 Mabsadd L
.67619 .68717 .56518 .7779 Mabsadd L
.67619 .68717 0 2.5 Mabsadd L closepath F
.67714 .68897 0 2.5 Mabsadd m
.67714 .68897 -0.56518 .7779 Mabsadd L
.67714 .68897 -2.37764 .77254 Mabsadd L
.67714 .68897 -0.91448 -0.29713 Mabsadd L
.67714 .68897 -1.46946 -2.02254 Mabsadd L
.67714 .68897 0 -0.96154 Mabsadd L
.67714 .68897 1.46946 -2.02254 Mabsadd L
.67714 .68897 .91448 -0.29713 Mabsadd L
.67714 .68897 2.37764 .77254 Mabsadd L
.67714 .68897 .56518 .7779 Mabsadd L
.67714 .68897 0 2.5 Mabsadd L closepath F
.6781 .68907 0 2.5 Mabsadd m
.6781 .68907 -0.56518 .7779 Mabsadd L
.6781 .68907 -2.37764 .77254 Mabsadd L
.6781 .68907 -0.91448 -0.29713 Mabsadd L
.6781 .68907 -1.46946 -2.02254 Mabsadd L
.6781 .68907 0 -0.96154 Mabsadd L
.6781 .68907 1.46946 -2.02254 Mabsadd L
.6781 .68907 .91448 -0.29713 Mabsadd L
.6781 .68907 2.37764 .77254 Mabsadd L
.6781 .68907 .56518 .7779 Mabsadd L
.6781 .68907 0 2.5 Mabsadd L closepath F
.67905 .68941 0 2.5 Mabsadd m
.67905 .68941 -0.56518 .7779 Mabsadd L
.67905 .68941 -2.37764 .77254 Mabsadd L
.67905 .68941 -0.91448 -0.29713 Mabsadd L
.67905 .68941 -1.46946 -2.02254 Mabsadd L
.67905 .68941 0 -0.96154 Mabsadd L
.67905 .68941 1.46946 -2.02254 Mabsadd L
.67905 .68941 .91448 -0.29713 Mabsadd L
.67905 .68941 2.37764 .77254 Mabsadd L
.67905 .68941 .56518 .7779 Mabsadd L
.67905 .68941 0 2.5 Mabsadd L closepath F
.68 .68944 0 2.5 Mabsadd m
.68 .68944 -0.56518 .7779 Mabsadd L
.68 .68944 -2.37764 .77254 Mabsadd L
.68 .68944 -0.91448 -0.29713 Mabsadd L
.68 .68944 -1.46946 -2.02254 Mabsadd L
.68 .68944 0 -0.96154 Mabsadd L
.68 .68944 1.46946 -2.02254 Mabsadd L
.68 .68944 .91448 -0.29713 Mabsadd L
.68 .68944 2.37764 .77254 Mabsadd L
.68 .68944 .56518 .7779 Mabsadd L
.68 .68944 0 2.5 Mabsadd L closepath F
.68095 .68952 0 2.5 Mabsadd m
.68095 .68952 -0.56518 .7779 Mabsadd L
.68095 .68952 -2.37764 .77254 Mabsadd L
.68095 .68952 -0.91448 -0.29713 Mabsadd L
.68095 .68952 -1.46946 -2.02254 Mabsadd L
.68095 .68952 0 -0.96154 Mabsadd L
.68095 .68952 1.46946 -2.02254 Mabsadd L
.68095 .68952 .91448 -0.29713 Mabsadd L
.68095 .68952 2.37764 .77254 Mabsadd L
.68095 .68952 .56518 .7779 Mabsadd L
.68095 .68952 0 2.5 Mabsadd L closepath F
.6819 .68952 0 2.5 Mabsadd m
.6819 .68952 -0.56518 .7779 Mabsadd L
.6819 .68952 -2.37764 .77254 Mabsadd L
.6819 .68952 -0.91448 -0.29713 Mabsadd L
.6819 .68952 -1.46946 -2.02254 Mabsadd L
.6819 .68952 0 -0.96154 Mabsadd L
.6819 .68952 1.46946 -2.02254 Mabsadd L
.6819 .68952 .91448 -0.29713 Mabsadd L
.6819 .68952 2.37764 .77254 Mabsadd L
.6819 .68952 .56518 .7779 Mabsadd L
.6819 .68952 0 2.5 Mabsadd L closepath F
.68286 .68955 0 2.5 Mabsadd m
.68286 .68955 -0.56518 .7779 Mabsadd L
.68286 .68955 -2.37764 .77254 Mabsadd L
.68286 .68955 -0.91448 -0.29713 Mabsadd L
.68286 .68955 -1.46946 -2.02254 Mabsadd L
.68286 .68955 0 -0.96154 Mabsadd L
.68286 .68955 1.46946 -2.02254 Mabsadd L
.68286 .68955 .91448 -0.29713 Mabsadd L
.68286 .68955 2.37764 .77254 Mabsadd L
.68286 .68955 .56518 .7779 Mabsadd L
.68286 .68955 0 2.5 Mabsadd L closepath F
.68381 .68959 0 2.5 Mabsadd m
.68381 .68959 -0.56518 .7779 Mabsadd L
.68381 .68959 -2.37764 .77254 Mabsadd L
.68381 .68959 -0.91448 -0.29713 Mabsadd L
.68381 .68959 -1.46946 -2.02254 Mabsadd L
.68381 .68959 0 -0.96154 Mabsadd L
.68381 .68959 1.46946 -2.02254 Mabsadd L
.68381 .68959 .91448 -0.29713 Mabsadd L
.68381 .68959 2.37764 .77254 Mabsadd L
.68381 .68959 .56518 .7779 Mabsadd L
.68381 .68959 0 2.5 Mabsadd L closepath F
.68476 .68959 0 2.5 Mabsadd m
.68476 .68959 -0.56518 .7779 Mabsadd L
.68476 .68959 -2.37764 .77254 Mabsadd L
.68476 .68959 -0.91448 -0.29713 Mabsadd L
.68476 .68959 -1.46946 -2.02254 Mabsadd L
.68476 .68959 0 -0.96154 Mabsadd L
.68476 .68959 1.46946 -2.02254 Mabsadd L
.68476 .68959 .91448 -0.29713 Mabsadd L
.68476 .68959 2.37764 .77254 Mabsadd L
.68476 .68959 .56518 .7779 Mabsadd L
.68476 .68959 0 2.5 Mabsadd L closepath F
.68571 .68959 0 2.5 Mabsadd m
.68571 .68959 -0.56518 .7779 Mabsadd L
.68571 .68959 -2.37764 .77254 Mabsadd L
.68571 .68959 -0.91448 -0.29713 Mabsadd L
.68571 .68959 -1.46946 -2.02254 Mabsadd L
.68571 .68959 0 -0.96154 Mabsadd L
.68571 .68959 1.46946 -2.02254 Mabsadd L
.68571 .68959 .91448 -0.29713 Mabsadd L
.68571 .68959 2.37764 .77254 Mabsadd L
.68571 .68959 .56518 .7779 Mabsadd L
.68571 .68959 0 2.5 Mabsadd L closepath F
.68667 .6896 0 2.5 Mabsadd m
.68667 .6896 -0.56518 .7779 Mabsadd L
.68667 .6896 -2.37764 .77254 Mabsadd L
.68667 .6896 -0.91448 -0.29713 Mabsadd L
.68667 .6896 -1.46946 -2.02254 Mabsadd L
.68667 .6896 0 -0.96154 Mabsadd L
.68667 .6896 1.46946 -2.02254 Mabsadd L
.68667 .6896 .91448 -0.29713 Mabsadd L
.68667 .6896 2.37764 .77254 Mabsadd L
.68667 .6896 .56518 .7779 Mabsadd L
.68667 .6896 0 2.5 Mabsadd L closepath F
.68762 .6896 0 2.5 Mabsadd m
.68762 .6896 -0.56518 .7779 Mabsadd L
.68762 .6896 -2.37764 .77254 Mabsadd L
.68762 .6896 -0.91448 -0.29713 Mabsadd L
.68762 .6896 -1.46946 -2.02254 Mabsadd L
.68762 .6896 0 -0.96154 Mabsadd L
.68762 .6896 1.46946 -2.02254 Mabsadd L
.68762 .6896 .91448 -0.29713 Mabsadd L
.68762 .6896 2.37764 .77254 Mabsadd L
.68762 .6896 .56518 .7779 Mabsadd L
.68762 .6896 0 2.5 Mabsadd L closepath F
.68857 .6896 0 2.5 Mabsadd m
.68857 .6896 -0.56518 .7779 Mabsadd L
.68857 .6896 -2.37764 .77254 Mabsadd L
.68857 .6896 -0.91448 -0.29713 Mabsadd L
.68857 .6896 -1.46946 -2.02254 Mabsadd L
.68857 .6896 0 -0.96154 Mabsadd L
.68857 .6896 1.46946 -2.02254 Mabsadd L
.68857 .6896 .91448 -0.29713 Mabsadd L
.68857 .6896 2.37764 .77254 Mabsadd L
.68857 .6896 .56518 .7779 Mabsadd L
.68857 .6896 0 2.5 Mabsadd L closepath F
.68952 .68961 0 2.5 Mabsadd m
.68952 .68961 -0.56518 .7779 Mabsadd L
.68952 .68961 -2.37764 .77254 Mabsadd L
.68952 .68961 -0.91448 -0.29713 Mabsadd L
.68952 .68961 -1.46946 -2.02254 Mabsadd L
.68952 .68961 0 -0.96154 Mabsadd L
.68952 .68961 1.46946 -2.02254 Mabsadd L
.68952 .68961 .91448 -0.29713 Mabsadd L
.68952 .68961 2.37764 .77254 Mabsadd L
.68952 .68961 .56518 .7779 Mabsadd L
.68952 .68961 0 2.5 Mabsadd L closepath F
.69048 .68961 0 2.5 Mabsadd m
.69048 .68961 -0.56518 .7779 Mabsadd L
.69048 .68961 -2.37764 .77254 Mabsadd L
.69048 .68961 -0.91448 -0.29713 Mabsadd L
.69048 .68961 -1.46946 -2.02254 Mabsadd L
.69048 .68961 0 -0.96154 Mabsadd L
.69048 .68961 1.46946 -2.02254 Mabsadd L
.69048 .68961 .91448 -0.29713 Mabsadd L
.69048 .68961 2.37764 .77254 Mabsadd L
.69048 .68961 .56518 .7779 Mabsadd L
.69048 .68961 0 2.5 Mabsadd L closepath F
.69143 .68961 0 2.5 Mabsadd m
.69143 .68961 -0.56518 .7779 Mabsadd L
.69143 .68961 -2.37764 .77254 Mabsadd L
.69143 .68961 -0.91448 -0.29713 Mabsadd L
.69143 .68961 -1.46946 -2.02254 Mabsadd L
.69143 .68961 0 -0.96154 Mabsadd L
.69143 .68961 1.46946 -2.02254 Mabsadd L
.69143 .68961 .91448 -0.29713 Mabsadd L
.69143 .68961 2.37764 .77254 Mabsadd L
.69143 .68961 .56518 .7779 Mabsadd L
.69143 .68961 0 2.5 Mabsadd L closepath F
.69238 .68961 0 2.5 Mabsadd m
.69238 .68961 -0.56518 .7779 Mabsadd L
.69238 .68961 -2.37764 .77254 Mabsadd L
.69238 .68961 -0.91448 -0.29713 Mabsadd L
.69238 .68961 -1.46946 -2.02254 Mabsadd L
.69238 .68961 0 -0.96154 Mabsadd L
.69238 .68961 1.46946 -2.02254 Mabsadd L
.69238 .68961 .91448 -0.29713 Mabsadd L
.69238 .68961 2.37764 .77254 Mabsadd L
.69238 .68961 .56518 .7779 Mabsadd L
.69238 .68961 0 2.5 Mabsadd L closepath F
.69333 .68961 0 2.5 Mabsadd m
.69333 .68961 -0.56518 .7779 Mabsadd L
.69333 .68961 -2.37764 .77254 Mabsadd L
.69333 .68961 -0.91448 -0.29713 Mabsadd L
.69333 .68961 -1.46946 -2.02254 Mabsadd L
.69333 .68961 0 -0.96154 Mabsadd L
.69333 .68961 1.46946 -2.02254 Mabsadd L
.69333 .68961 .91448 -0.29713 Mabsadd L
.69333 .68961 2.37764 .77254 Mabsadd L
.69333 .68961 .56518 .7779 Mabsadd L
.69333 .68961 0 2.5 Mabsadd L closepath F
.69429 .68961 0 2.5 Mabsadd m
.69429 .68961 -0.56518 .7779 Mabsadd L
.69429 .68961 -2.37764 .77254 Mabsadd L
.69429 .68961 -0.91448 -0.29713 Mabsadd L
.69429 .68961 -1.46946 -2.02254 Mabsadd L
.69429 .68961 0 -0.96154 Mabsadd L
.69429 .68961 1.46946 -2.02254 Mabsadd L
.69429 .68961 .91448 -0.29713 Mabsadd L
.69429 .68961 2.37764 .77254 Mabsadd L
.69429 .68961 .56518 .7779 Mabsadd L
.69429 .68961 0 2.5 Mabsadd L closepath F
.69524 .6896 0 2.5 Mabsadd m
.69524 .6896 -0.56518 .7779 Mabsadd L
.69524 .6896 -2.37764 .77254 Mabsadd L
.69524 .6896 -0.91448 -0.29713 Mabsadd L
.69524 .6896 -1.46946 -2.02254 Mabsadd L
.69524 .6896 0 -0.96154 Mabsadd L
.69524 .6896 1.46946 -2.02254 Mabsadd L
.69524 .6896 .91448 -0.29713 Mabsadd L
.69524 .6896 2.37764 .77254 Mabsadd L
.69524 .6896 .56518 .7779 Mabsadd L
.69524 .6896 0 2.5 Mabsadd L closepath F
.69619 .6896 0 2.5 Mabsadd m
.69619 .6896 -0.56518 .7779 Mabsadd L
.69619 .6896 -2.37764 .77254 Mabsadd L
.69619 .6896 -0.91448 -0.29713 Mabsadd L
.69619 .6896 -1.46946 -2.02254 Mabsadd L
.69619 .6896 0 -0.96154 Mabsadd L
.69619 .6896 1.46946 -2.02254 Mabsadd L
.69619 .6896 .91448 -0.29713 Mabsadd L
.69619 .6896 2.37764 .77254 Mabsadd L
.69619 .6896 .56518 .7779 Mabsadd L
.69619 .6896 0 2.5 Mabsadd L closepath F
.69714 .6896 0 2.5 Mabsadd m
.69714 .6896 -0.56518 .7779 Mabsadd L
.69714 .6896 -2.37764 .77254 Mabsadd L
.69714 .6896 -0.91448 -0.29713 Mabsadd L
.69714 .6896 -1.46946 -2.02254 Mabsadd L
.69714 .6896 0 -0.96154 Mabsadd L
.69714 .6896 1.46946 -2.02254 Mabsadd L
.69714 .6896 .91448 -0.29713 Mabsadd L
.69714 .6896 2.37764 .77254 Mabsadd L
.69714 .6896 .56518 .7779 Mabsadd L
.69714 .6896 0 2.5 Mabsadd L closepath F
.6981 .6896 0 2.5 Mabsadd m
.6981 .6896 -0.56518 .7779 Mabsadd L
.6981 .6896 -2.37764 .77254 Mabsadd L
.6981 .6896 -0.91448 -0.29713 Mabsadd L
.6981 .6896 -1.46946 -2.02254 Mabsadd L
.6981 .6896 0 -0.96154 Mabsadd L
.6981 .6896 1.46946 -2.02254 Mabsadd L
.6981 .6896 .91448 -0.29713 Mabsadd L
.6981 .6896 2.37764 .77254 Mabsadd L
.6981 .6896 .56518 .7779 Mabsadd L
.6981 .6896 0 2.5 Mabsadd L closepath F
.69905 .68959 0 2.5 Mabsadd m
.69905 .68959 -0.56518 .7779 Mabsadd L
.69905 .68959 -2.37764 .77254 Mabsadd L
.69905 .68959 -0.91448 -0.29713 Mabsadd L
.69905 .68959 -1.46946 -2.02254 Mabsadd L
.69905 .68959 0 -0.96154 Mabsadd L
.69905 .68959 1.46946 -2.02254 Mabsadd L
.69905 .68959 .91448 -0.29713 Mabsadd L
.69905 .68959 2.37764 .77254 Mabsadd L
.69905 .68959 .56518 .7779 Mabsadd L
.69905 .68959 0 2.5 Mabsadd L closepath F
.7 .68959 0 2.5 Mabsadd m
.7 .68959 -0.56518 .7779 Mabsadd L
.7 .68959 -2.37764 .77254 Mabsadd L
.7 .68959 -0.91448 -0.29713 Mabsadd L
.7 .68959 -1.46946 -2.02254 Mabsadd L
.7 .68959 0 -0.96154 Mabsadd L
.7 .68959 1.46946 -2.02254 Mabsadd L
.7 .68959 .91448 -0.29713 Mabsadd L
.7 .68959 2.37764 .77254 Mabsadd L
.7 .68959 .56518 .7779 Mabsadd L
.7 .68959 0 2.5 Mabsadd L closepath F
.70095 .68958 0 2.5 Mabsadd m
.70095 .68958 -0.56518 .7779 Mabsadd L
.70095 .68958 -2.37764 .77254 Mabsadd L
.70095 .68958 -0.91448 -0.29713 Mabsadd L
.70095 .68958 -1.46946 -2.02254 Mabsadd L
.70095 .68958 0 -0.96154 Mabsadd L
.70095 .68958 1.46946 -2.02254 Mabsadd L
.70095 .68958 .91448 -0.29713 Mabsadd L
.70095 .68958 2.37764 .77254 Mabsadd L
.70095 .68958 .56518 .7779 Mabsadd L
.70095 .68958 0 2.5 Mabsadd L closepath F
.7019 .68957 0 2.5 Mabsadd m
.7019 .68957 -0.56518 .7779 Mabsadd L
.7019 .68957 -2.37764 .77254 Mabsadd L
.7019 .68957 -0.91448 -0.29713 Mabsadd L
.7019 .68957 -1.46946 -2.02254 Mabsadd L
.7019 .68957 0 -0.96154 Mabsadd L
.7019 .68957 1.46946 -2.02254 Mabsadd L
.7019 .68957 .91448 -0.29713 Mabsadd L
.7019 .68957 2.37764 .77254 Mabsadd L
.7019 .68957 .56518 .7779 Mabsadd L
.7019 .68957 0 2.5 Mabsadd L closepath F
.70286 .68956 0 2.5 Mabsadd m
.70286 .68956 -0.56518 .7779 Mabsadd L
.70286 .68956 -2.37764 .77254 Mabsadd L
.70286 .68956 -0.91448 -0.29713 Mabsadd L
.70286 .68956 -1.46946 -2.02254 Mabsadd L
.70286 .68956 0 -0.96154 Mabsadd L
.70286 .68956 1.46946 -2.02254 Mabsadd L
.70286 .68956 .91448 -0.29713 Mabsadd L
.70286 .68956 2.37764 .77254 Mabsadd L
.70286 .68956 .56518 .7779 Mabsadd L
.70286 .68956 0 2.5 Mabsadd L closepath F
.70381 .68954 0 2.5 Mabsadd m
.70381 .68954 -0.56518 .7779 Mabsadd L
.70381 .68954 -2.37764 .77254 Mabsadd L
.70381 .68954 -0.91448 -0.29713 Mabsadd L
.70381 .68954 -1.46946 -2.02254 Mabsadd L
.70381 .68954 0 -0.96154 Mabsadd L
.70381 .68954 1.46946 -2.02254 Mabsadd L
.70381 .68954 .91448 -0.29713 Mabsadd L
.70381 .68954 2.37764 .77254 Mabsadd L
.70381 .68954 .56518 .7779 Mabsadd L
.70381 .68954 0 2.5 Mabsadd L closepath F
.70476 .68945 0 2.5 Mabsadd m
.70476 .68945 -0.56518 .7779 Mabsadd L
.70476 .68945 -2.37764 .77254 Mabsadd L
.70476 .68945 -0.91448 -0.29713 Mabsadd L
.70476 .68945 -1.46946 -2.02254 Mabsadd L
.70476 .68945 0 -0.96154 Mabsadd L
.70476 .68945 1.46946 -2.02254 Mabsadd L
.70476 .68945 .91448 -0.29713 Mabsadd L
.70476 .68945 2.37764 .77254 Mabsadd L
.70476 .68945 .56518 .7779 Mabsadd L
.70476 .68945 0 2.5 Mabsadd L closepath F
.70571 .68943 0 2.5 Mabsadd m
.70571 .68943 -0.56518 .7779 Mabsadd L
.70571 .68943 -2.37764 .77254 Mabsadd L
.70571 .68943 -0.91448 -0.29713 Mabsadd L
.70571 .68943 -1.46946 -2.02254 Mabsadd L
.70571 .68943 0 -0.96154 Mabsadd L
.70571 .68943 1.46946 -2.02254 Mabsadd L
.70571 .68943 .91448 -0.29713 Mabsadd L
.70571 .68943 2.37764 .77254 Mabsadd L
.70571 .68943 .56518 .7779 Mabsadd L
.70571 .68943 0 2.5 Mabsadd L closepath F
.70667 .68926 0 2.5 Mabsadd m
.70667 .68926 -0.56518 .7779 Mabsadd L
.70667 .68926 -2.37764 .77254 Mabsadd L
.70667 .68926 -0.91448 -0.29713 Mabsadd L
.70667 .68926 -1.46946 -2.02254 Mabsadd L
.70667 .68926 0 -0.96154 Mabsadd L
.70667 .68926 1.46946 -2.02254 Mabsadd L
.70667 .68926 .91448 -0.29713 Mabsadd L
.70667 .68926 2.37764 .77254 Mabsadd L
.70667 .68926 .56518 .7779 Mabsadd L
.70667 .68926 0 2.5 Mabsadd L closepath F
.70762 .68918 0 2.5 Mabsadd m
.70762 .68918 -0.56518 .7779 Mabsadd L
.70762 .68918 -2.37764 .77254 Mabsadd L
.70762 .68918 -0.91448 -0.29713 Mabsadd L
.70762 .68918 -1.46946 -2.02254 Mabsadd L
.70762 .68918 0 -0.96154 Mabsadd L
.70762 .68918 1.46946 -2.02254 Mabsadd L
.70762 .68918 .91448 -0.29713 Mabsadd L
.70762 .68918 2.37764 .77254 Mabsadd L
.70762 .68918 .56518 .7779 Mabsadd L
.70762 .68918 0 2.5 Mabsadd L closepath F
.70857 .68837 0 2.5 Mabsadd m
.70857 .68837 -0.56518 .7779 Mabsadd L
.70857 .68837 -2.37764 .77254 Mabsadd L
.70857 .68837 -0.91448 -0.29713 Mabsadd L
.70857 .68837 -1.46946 -2.02254 Mabsadd L
.70857 .68837 0 -0.96154 Mabsadd L
.70857 .68837 1.46946 -2.02254 Mabsadd L
.70857 .68837 .91448 -0.29713 Mabsadd L
.70857 .68837 2.37764 .77254 Mabsadd L
.70857 .68837 .56518 .7779 Mabsadd L
.70857 .68837 0 2.5 Mabsadd L closepath F
.70952 .6879 0 2.5 Mabsadd m
.70952 .6879 -0.56518 .7779 Mabsadd L
.70952 .6879 -2.37764 .77254 Mabsadd L
.70952 .6879 -0.91448 -0.29713 Mabsadd L
.70952 .6879 -1.46946 -2.02254 Mabsadd L
.70952 .6879 0 -0.96154 Mabsadd L
.70952 .6879 1.46946 -2.02254 Mabsadd L
.70952 .6879 .91448 -0.29713 Mabsadd L
.70952 .6879 2.37764 .77254 Mabsadd L
.70952 .6879 .56518 .7779 Mabsadd L
.70952 .6879 0 2.5 Mabsadd L closepath F
.71048 .68665 0 2.5 Mabsadd m
.71048 .68665 -0.56518 .7779 Mabsadd L
.71048 .68665 -2.37764 .77254 Mabsadd L
.71048 .68665 -0.91448 -0.29713 Mabsadd L
.71048 .68665 -1.46946 -2.02254 Mabsadd L
.71048 .68665 0 -0.96154 Mabsadd L
.71048 .68665 1.46946 -2.02254 Mabsadd L
.71048 .68665 .91448 -0.29713 Mabsadd L
.71048 .68665 2.37764 .77254 Mabsadd L
.71048 .68665 .56518 .7779 Mabsadd L
.71048 .68665 0 2.5 Mabsadd L closepath F
.71143 .68293 0 2.5 Mabsadd m
.71143 .68293 -0.56518 .7779 Mabsadd L
.71143 .68293 -2.37764 .77254 Mabsadd L
.71143 .68293 -0.91448 -0.29713 Mabsadd L
.71143 .68293 -1.46946 -2.02254 Mabsadd L
.71143 .68293 0 -0.96154 Mabsadd L
.71143 .68293 1.46946 -2.02254 Mabsadd L
.71143 .68293 .91448 -0.29713 Mabsadd L
.71143 .68293 2.37764 .77254 Mabsadd L
.71143 .68293 .56518 .7779 Mabsadd L
.71143 .68293 0 2.5 Mabsadd L closepath F
.71238 .67078 0 2.5 Mabsadd m
.71238 .67078 -0.56518 .7779 Mabsadd L
.71238 .67078 -2.37764 .77254 Mabsadd L
.71238 .67078 -0.91448 -0.29713 Mabsadd L
.71238 .67078 -1.46946 -2.02254 Mabsadd L
.71238 .67078 0 -0.96154 Mabsadd L
.71238 .67078 1.46946 -2.02254 Mabsadd L
.71238 .67078 .91448 -0.29713 Mabsadd L
.71238 .67078 2.37764 .77254 Mabsadd L
.71238 .67078 .56518 .7779 Mabsadd L
.71238 .67078 0 2.5 Mabsadd L closepath F
.71333 .67078 0 2.5 Mabsadd m
.71333 .67078 -0.56518 .7779 Mabsadd L
.71333 .67078 -2.37764 .77254 Mabsadd L
.71333 .67078 -0.91448 -0.29713 Mabsadd L
.71333 .67078 -1.46946 -2.02254 Mabsadd L
.71333 .67078 0 -0.96154 Mabsadd L
.71333 .67078 1.46946 -2.02254 Mabsadd L
.71333 .67078 .91448 -0.29713 Mabsadd L
.71333 .67078 2.37764 .77254 Mabsadd L
.71333 .67078 .56518 .7779 Mabsadd L
.71333 .67078 0 2.5 Mabsadd L closepath F
.71429 .67099 0 2.5 Mabsadd m
.71429 .67099 -0.56518 .7779 Mabsadd L
.71429 .67099 -2.37764 .77254 Mabsadd L
.71429 .67099 -0.91448 -0.29713 Mabsadd L
.71429 .67099 -1.46946 -2.02254 Mabsadd L
.71429 .67099 0 -0.96154 Mabsadd L
.71429 .67099 1.46946 -2.02254 Mabsadd L
.71429 .67099 .91448 -0.29713 Mabsadd L
.71429 .67099 2.37764 .77254 Mabsadd L
.71429 .67099 .56518 .7779 Mabsadd L
.71429 .67099 0 2.5 Mabsadd L closepath F
.71524 .67524 0 2.5 Mabsadd m
.71524 .67524 -0.56518 .7779 Mabsadd L
.71524 .67524 -2.37764 .77254 Mabsadd L
.71524 .67524 -0.91448 -0.29713 Mabsadd L
.71524 .67524 -1.46946 -2.02254 Mabsadd L
.71524 .67524 0 -0.96154 Mabsadd L
.71524 .67524 1.46946 -2.02254 Mabsadd L
.71524 .67524 .91448 -0.29713 Mabsadd L
.71524 .67524 2.37764 .77254 Mabsadd L
.71524 .67524 .56518 .7779 Mabsadd L
.71524 .67524 0 2.5 Mabsadd L closepath F
.71619 .72164 0 2.5 Mabsadd m
.71619 .72164 -0.56518 .7779 Mabsadd L
.71619 .72164 -2.37764 .77254 Mabsadd L
.71619 .72164 -0.91448 -0.29713 Mabsadd L
.71619 .72164 -1.46946 -2.02254 Mabsadd L
.71619 .72164 0 -0.96154 Mabsadd L
.71619 .72164 1.46946 -2.02254 Mabsadd L
.71619 .72164 .91448 -0.29713 Mabsadd L
.71619 .72164 2.37764 .77254 Mabsadd L
.71619 .72164 .56518 .7779 Mabsadd L
.71619 .72164 0 2.5 Mabsadd L closepath F
.71714 .77367 0 2.5 Mabsadd m
.71714 .77367 -0.56518 .7779 Mabsadd L
.71714 .77367 -2.37764 .77254 Mabsadd L
.71714 .77367 -0.91448 -0.29713 Mabsadd L
.71714 .77367 -1.46946 -2.02254 Mabsadd L
.71714 .77367 0 -0.96154 Mabsadd L
.71714 .77367 1.46946 -2.02254 Mabsadd L
.71714 .77367 .91448 -0.29713 Mabsadd L
.71714 .77367 2.37764 .77254 Mabsadd L
.71714 .77367 .56518 .7779 Mabsadd L
.71714 .77367 0 2.5 Mabsadd L closepath F
.7181 .77407 0 2.5 Mabsadd m
.7181 .77407 -0.56518 .7779 Mabsadd L
.7181 .77407 -2.37764 .77254 Mabsadd L
.7181 .77407 -0.91448 -0.29713 Mabsadd L
.7181 .77407 -1.46946 -2.02254 Mabsadd L
.7181 .77407 0 -0.96154 Mabsadd L
.7181 .77407 1.46946 -2.02254 Mabsadd L
.7181 .77407 .91448 -0.29713 Mabsadd L
.7181 .77407 2.37764 .77254 Mabsadd L
.7181 .77407 .56518 .7779 Mabsadd L
.7181 .77407 0 2.5 Mabsadd L closepath F
.71905 .77709 0 2.5 Mabsadd m
.71905 .77709 -0.56518 .7779 Mabsadd L
.71905 .77709 -2.37764 .77254 Mabsadd L
.71905 .77709 -0.91448 -0.29713 Mabsadd L
.71905 .77709 -1.46946 -2.02254 Mabsadd L
.71905 .77709 0 -0.96154 Mabsadd L
.71905 .77709 1.46946 -2.02254 Mabsadd L
.71905 .77709 .91448 -0.29713 Mabsadd L
.71905 .77709 2.37764 .77254 Mabsadd L
.71905 .77709 .56518 .7779 Mabsadd L
.71905 .77709 0 2.5 Mabsadd L closepath F
.72 .77993 0 2.5 Mabsadd m
.72 .77993 -0.56518 .7779 Mabsadd L
.72 .77993 -2.37764 .77254 Mabsadd L
.72 .77993 -0.91448 -0.29713 Mabsadd L
.72 .77993 -1.46946 -2.02254 Mabsadd L
.72 .77993 0 -0.96154 Mabsadd L
.72 .77993 1.46946 -2.02254 Mabsadd L
.72 .77993 .91448 -0.29713 Mabsadd L
.72 .77993 2.37764 .77254 Mabsadd L
.72 .77993 .56518 .7779 Mabsadd L
.72 .77993 0 2.5 Mabsadd L closepath F
.72095 .77971 0 2.5 Mabsadd m
.72095 .77971 -0.56518 .7779 Mabsadd L
.72095 .77971 -2.37764 .77254 Mabsadd L
.72095 .77971 -0.91448 -0.29713 Mabsadd L
.72095 .77971 -1.46946 -2.02254 Mabsadd L
.72095 .77971 0 -0.96154 Mabsadd L
.72095 .77971 1.46946 -2.02254 Mabsadd L
.72095 .77971 .91448 -0.29713 Mabsadd L
.72095 .77971 2.37764 .77254 Mabsadd L
.72095 .77971 .56518 .7779 Mabsadd L
.72095 .77971 0 2.5 Mabsadd L closepath F
.7219 .84103 0 2.5 Mabsadd m
.7219 .84103 -0.56518 .7779 Mabsadd L
.7219 .84103 -2.37764 .77254 Mabsadd L
.7219 .84103 -0.91448 -0.29713 Mabsadd L
.7219 .84103 -1.46946 -2.02254 Mabsadd L
.7219 .84103 0 -0.96154 Mabsadd L
.7219 .84103 1.46946 -2.02254 Mabsadd L
.7219 .84103 .91448 -0.29713 Mabsadd L
.7219 .84103 2.37764 .77254 Mabsadd L
.7219 .84103 .56518 .7779 Mabsadd L
.7219 .84103 0 2.5 Mabsadd L closepath F
.72286 .85076 0 2.5 Mabsadd m
.72286 .85076 -0.56518 .7779 Mabsadd L
.72286 .85076 -2.37764 .77254 Mabsadd L
.72286 .85076 -0.91448 -0.29713 Mabsadd L
.72286 .85076 -1.46946 -2.02254 Mabsadd L
.72286 .85076 0 -0.96154 Mabsadd L
.72286 .85076 1.46946 -2.02254 Mabsadd L
.72286 .85076 .91448 -0.29713 Mabsadd L
.72286 .85076 2.37764 .77254 Mabsadd L
.72286 .85076 .56518 .7779 Mabsadd L
.72286 .85076 0 2.5 Mabsadd L closepath F
.72381 .85159 0 2.5 Mabsadd m
.72381 .85159 -0.56518 .7779 Mabsadd L
.72381 .85159 -2.37764 .77254 Mabsadd L
.72381 .85159 -0.91448 -0.29713 Mabsadd L
.72381 .85159 -1.46946 -2.02254 Mabsadd L
.72381 .85159 0 -0.96154 Mabsadd L
.72381 .85159 1.46946 -2.02254 Mabsadd L
.72381 .85159 .91448 -0.29713 Mabsadd L
.72381 .85159 2.37764 .77254 Mabsadd L
.72381 .85159 .56518 .7779 Mabsadd L
.72381 .85159 0 2.5 Mabsadd L closepath F
.72476 .854 0 2.5 Mabsadd m
.72476 .854 -0.56518 .7779 Mabsadd L
.72476 .854 -2.37764 .77254 Mabsadd L
.72476 .854 -0.91448 -0.29713 Mabsadd L
.72476 .854 -1.46946 -2.02254 Mabsadd L
.72476 .854 0 -0.96154 Mabsadd L
.72476 .854 1.46946 -2.02254 Mabsadd L
.72476 .854 .91448 -0.29713 Mabsadd L
.72476 .854 2.37764 .77254 Mabsadd L
.72476 .854 .56518 .7779 Mabsadd L
.72476 .854 0 2.5 Mabsadd L closepath F
.72571 .85517 0 2.5 Mabsadd m
.72571 .85517 -0.56518 .7779 Mabsadd L
.72571 .85517 -2.37764 .77254 Mabsadd L
.72571 .85517 -0.91448 -0.29713 Mabsadd L
.72571 .85517 -1.46946 -2.02254 Mabsadd L
.72571 .85517 0 -0.96154 Mabsadd L
.72571 .85517 1.46946 -2.02254 Mabsadd L
.72571 .85517 .91448 -0.29713 Mabsadd L
.72571 .85517 2.37764 .77254 Mabsadd L
.72571 .85517 .56518 .7779 Mabsadd L
.72571 .85517 0 2.5 Mabsadd L closepath F
.72667 .85488 0 2.5 Mabsadd m
.72667 .85488 -0.56518 .7779 Mabsadd L
.72667 .85488 -2.37764 .77254 Mabsadd L
.72667 .85488 -0.91448 -0.29713 Mabsadd L
.72667 .85488 -1.46946 -2.02254 Mabsadd L
.72667 .85488 0 -0.96154 Mabsadd L
.72667 .85488 1.46946 -2.02254 Mabsadd L
.72667 .85488 .91448 -0.29713 Mabsadd L
.72667 .85488 2.37764 .77254 Mabsadd L
.72667 .85488 .56518 .7779 Mabsadd L
.72667 .85488 0 2.5 Mabsadd L closepath F
.72762 .85995 0 2.5 Mabsadd m
.72762 .85995 -0.56518 .7779 Mabsadd L
.72762 .85995 -2.37764 .77254 Mabsadd L
.72762 .85995 -0.91448 -0.29713 Mabsadd L
.72762 .85995 -1.46946 -2.02254 Mabsadd L
.72762 .85995 0 -0.96154 Mabsadd L
.72762 .85995 1.46946 -2.02254 Mabsadd L
.72762 .85995 .91448 -0.29713 Mabsadd L
.72762 .85995 2.37764 .77254 Mabsadd L
.72762 .85995 .56518 .7779 Mabsadd L
.72762 .85995 0 2.5 Mabsadd L closepath F
.72857 .86174 0 2.5 Mabsadd m
.72857 .86174 -0.56518 .7779 Mabsadd L
.72857 .86174 -2.37764 .77254 Mabsadd L
.72857 .86174 -0.91448 -0.29713 Mabsadd L
.72857 .86174 -1.46946 -2.02254 Mabsadd L
.72857 .86174 0 -0.96154 Mabsadd L
.72857 .86174 1.46946 -2.02254 Mabsadd L
.72857 .86174 .91448 -0.29713 Mabsadd L
.72857 .86174 2.37764 .77254 Mabsadd L
.72857 .86174 .56518 .7779 Mabsadd L
.72857 .86174 0 2.5 Mabsadd L closepath F
.72952 .867 0 2.5 Mabsadd m
.72952 .867 -0.56518 .7779 Mabsadd L
.72952 .867 -2.37764 .77254 Mabsadd L
.72952 .867 -0.91448 -0.29713 Mabsadd L
.72952 .867 -1.46946 -2.02254 Mabsadd L
.72952 .867 0 -0.96154 Mabsadd L
.72952 .867 1.46946 -2.02254 Mabsadd L
.72952 .867 .91448 -0.29713 Mabsadd L
.72952 .867 2.37764 .77254 Mabsadd L
.72952 .867 .56518 .7779 Mabsadd L
.72952 .867 0 2.5 Mabsadd L closepath F
.73048 .86727 0 2.5 Mabsadd m
.73048 .86727 -0.56518 .7779 Mabsadd L
.73048 .86727 -2.37764 .77254 Mabsadd L
.73048 .86727 -0.91448 -0.29713 Mabsadd L
.73048 .86727 -1.46946 -2.02254 Mabsadd L
.73048 .86727 0 -0.96154 Mabsadd L
.73048 .86727 1.46946 -2.02254 Mabsadd L
.73048 .86727 .91448 -0.29713 Mabsadd L
.73048 .86727 2.37764 .77254 Mabsadd L
.73048 .86727 .56518 .7779 Mabsadd L
.73048 .86727 0 2.5 Mabsadd L closepath F
.73143 .86811 0 2.5 Mabsadd m
.73143 .86811 -0.56518 .7779 Mabsadd L
.73143 .86811 -2.37764 .77254 Mabsadd L
.73143 .86811 -0.91448 -0.29713 Mabsadd L
.73143 .86811 -1.46946 -2.02254 Mabsadd L
.73143 .86811 0 -0.96154 Mabsadd L
.73143 .86811 1.46946 -2.02254 Mabsadd L
.73143 .86811 .91448 -0.29713 Mabsadd L
.73143 .86811 2.37764 .77254 Mabsadd L
.73143 .86811 .56518 .7779 Mabsadd L
.73143 .86811 0 2.5 Mabsadd L closepath F
.73238 .86816 0 2.5 Mabsadd m
.73238 .86816 -0.56518 .7779 Mabsadd L
.73238 .86816 -2.37764 .77254 Mabsadd L
.73238 .86816 -0.91448 -0.29713 Mabsadd L
.73238 .86816 -1.46946 -2.02254 Mabsadd L
.73238 .86816 0 -0.96154 Mabsadd L
.73238 .86816 1.46946 -2.02254 Mabsadd L
.73238 .86816 .91448 -0.29713 Mabsadd L
.73238 .86816 2.37764 .77254 Mabsadd L
.73238 .86816 .56518 .7779 Mabsadd L
.73238 .86816 0 2.5 Mabsadd L closepath F
.73333 .86834 0 2.5 Mabsadd m
.73333 .86834 -0.56518 .7779 Mabsadd L
.73333 .86834 -2.37764 .77254 Mabsadd L
.73333 .86834 -0.91448 -0.29713 Mabsadd L
.73333 .86834 -1.46946 -2.02254 Mabsadd L
.73333 .86834 0 -0.96154 Mabsadd L
.73333 .86834 1.46946 -2.02254 Mabsadd L
.73333 .86834 .91448 -0.29713 Mabsadd L
.73333 .86834 2.37764 .77254 Mabsadd L
.73333 .86834 .56518 .7779 Mabsadd L
.73333 .86834 0 2.5 Mabsadd L closepath F
.73429 .86835 0 2.5 Mabsadd m
.73429 .86835 -0.56518 .7779 Mabsadd L
.73429 .86835 -2.37764 .77254 Mabsadd L
.73429 .86835 -0.91448 -0.29713 Mabsadd L
.73429 .86835 -1.46946 -2.02254 Mabsadd L
.73429 .86835 0 -0.96154 Mabsadd L
.73429 .86835 1.46946 -2.02254 Mabsadd L
.73429 .86835 .91448 -0.29713 Mabsadd L
.73429 .86835 2.37764 .77254 Mabsadd L
.73429 .86835 .56518 .7779 Mabsadd L
.73429 .86835 0 2.5 Mabsadd L closepath F
.73524 .8684 0 2.5 Mabsadd m
.73524 .8684 -0.56518 .7779 Mabsadd L
.73524 .8684 -2.37764 .77254 Mabsadd L
.73524 .8684 -0.91448 -0.29713 Mabsadd L
.73524 .8684 -1.46946 -2.02254 Mabsadd L
.73524 .8684 0 -0.96154 Mabsadd L
.73524 .8684 1.46946 -2.02254 Mabsadd L
.73524 .8684 .91448 -0.29713 Mabsadd L
.73524 .8684 2.37764 .77254 Mabsadd L
.73524 .8684 .56518 .7779 Mabsadd L
.73524 .8684 0 2.5 Mabsadd L closepath F
.73619 .8684 0 2.5 Mabsadd m
.73619 .8684 -0.56518 .7779 Mabsadd L
.73619 .8684 -2.37764 .77254 Mabsadd L
.73619 .8684 -0.91448 -0.29713 Mabsadd L
.73619 .8684 -1.46946 -2.02254 Mabsadd L
.73619 .8684 0 -0.96154 Mabsadd L
.73619 .8684 1.46946 -2.02254 Mabsadd L
.73619 .8684 .91448 -0.29713 Mabsadd L
.73619 .8684 2.37764 .77254 Mabsadd L
.73619 .8684 .56518 .7779 Mabsadd L
.73619 .8684 0 2.5 Mabsadd L closepath F
.73714 .86841 0 2.5 Mabsadd m
.73714 .86841 -0.56518 .7779 Mabsadd L
.73714 .86841 -2.37764 .77254 Mabsadd L
.73714 .86841 -0.91448 -0.29713 Mabsadd L
.73714 .86841 -1.46946 -2.02254 Mabsadd L
.73714 .86841 0 -0.96154 Mabsadd L
.73714 .86841 1.46946 -2.02254 Mabsadd L
.73714 .86841 .91448 -0.29713 Mabsadd L
.73714 .86841 2.37764 .77254 Mabsadd L
.73714 .86841 .56518 .7779 Mabsadd L
.73714 .86841 0 2.5 Mabsadd L closepath F
.7381 .86844 0 2.5 Mabsadd m
.7381 .86844 -0.56518 .7779 Mabsadd L
.7381 .86844 -2.37764 .77254 Mabsadd L
.7381 .86844 -0.91448 -0.29713 Mabsadd L
.7381 .86844 -1.46946 -2.02254 Mabsadd L
.7381 .86844 0 -0.96154 Mabsadd L
.7381 .86844 1.46946 -2.02254 Mabsadd L
.7381 .86844 .91448 -0.29713 Mabsadd L
.7381 .86844 2.37764 .77254 Mabsadd L
.7381 .86844 .56518 .7779 Mabsadd L
.7381 .86844 0 2.5 Mabsadd L closepath F
.73905 .86844 0 2.5 Mabsadd m
.73905 .86844 -0.56518 .7779 Mabsadd L
.73905 .86844 -2.37764 .77254 Mabsadd L
.73905 .86844 -0.91448 -0.29713 Mabsadd L
.73905 .86844 -1.46946 -2.02254 Mabsadd L
.73905 .86844 0 -0.96154 Mabsadd L
.73905 .86844 1.46946 -2.02254 Mabsadd L
.73905 .86844 .91448 -0.29713 Mabsadd L
.73905 .86844 2.37764 .77254 Mabsadd L
.73905 .86844 .56518 .7779 Mabsadd L
.73905 .86844 0 2.5 Mabsadd L closepath F
.74 .86844 0 2.5 Mabsadd m
.74 .86844 -0.56518 .7779 Mabsadd L
.74 .86844 -2.37764 .77254 Mabsadd L
.74 .86844 -0.91448 -0.29713 Mabsadd L
.74 .86844 -1.46946 -2.02254 Mabsadd L
.74 .86844 0 -0.96154 Mabsadd L
.74 .86844 1.46946 -2.02254 Mabsadd L
.74 .86844 .91448 -0.29713 Mabsadd L
.74 .86844 2.37764 .77254 Mabsadd L
.74 .86844 .56518 .7779 Mabsadd L
.74 .86844 0 2.5 Mabsadd L closepath F
.74095 .86845 0 2.5 Mabsadd m
.74095 .86845 -0.56518 .7779 Mabsadd L
.74095 .86845 -2.37764 .77254 Mabsadd L
.74095 .86845 -0.91448 -0.29713 Mabsadd L
.74095 .86845 -1.46946 -2.02254 Mabsadd L
.74095 .86845 0 -0.96154 Mabsadd L
.74095 .86845 1.46946 -2.02254 Mabsadd L
.74095 .86845 .91448 -0.29713 Mabsadd L
.74095 .86845 2.37764 .77254 Mabsadd L
.74095 .86845 .56518 .7779 Mabsadd L
.74095 .86845 0 2.5 Mabsadd L closepath F
.7419 .86846 0 2.5 Mabsadd m
.7419 .86846 -0.56518 .7779 Mabsadd L
.7419 .86846 -2.37764 .77254 Mabsadd L
.7419 .86846 -0.91448 -0.29713 Mabsadd L
.7419 .86846 -1.46946 -2.02254 Mabsadd L
.7419 .86846 0 -0.96154 Mabsadd L
.7419 .86846 1.46946 -2.02254 Mabsadd L
.7419 .86846 .91448 -0.29713 Mabsadd L
.7419 .86846 2.37764 .77254 Mabsadd L
.7419 .86846 .56518 .7779 Mabsadd L
.7419 .86846 0 2.5 Mabsadd L closepath F
.74286 .86846 0 2.5 Mabsadd m
.74286 .86846 -0.56518 .7779 Mabsadd L
.74286 .86846 -2.37764 .77254 Mabsadd L
.74286 .86846 -0.91448 -0.29713 Mabsadd L
.74286 .86846 -1.46946 -2.02254 Mabsadd L
.74286 .86846 0 -0.96154 Mabsadd L
.74286 .86846 1.46946 -2.02254 Mabsadd L
.74286 .86846 .91448 -0.29713 Mabsadd L
.74286 .86846 2.37764 .77254 Mabsadd L
.74286 .86846 .56518 .7779 Mabsadd L
.74286 .86846 0 2.5 Mabsadd L closepath F
.74381 .86846 0 2.5 Mabsadd m
.74381 .86846 -0.56518 .7779 Mabsadd L
.74381 .86846 -2.37764 .77254 Mabsadd L
.74381 .86846 -0.91448 -0.29713 Mabsadd L
.74381 .86846 -1.46946 -2.02254 Mabsadd L
.74381 .86846 0 -0.96154 Mabsadd L
.74381 .86846 1.46946 -2.02254 Mabsadd L
.74381 .86846 .91448 -0.29713 Mabsadd L
.74381 .86846 2.37764 .77254 Mabsadd L
.74381 .86846 .56518 .7779 Mabsadd L
.74381 .86846 0 2.5 Mabsadd L closepath F
.74476 .86846 0 2.5 Mabsadd m
.74476 .86846 -0.56518 .7779 Mabsadd L
.74476 .86846 -2.37764 .77254 Mabsadd L
.74476 .86846 -0.91448 -0.29713 Mabsadd L
.74476 .86846 -1.46946 -2.02254 Mabsadd L
.74476 .86846 0 -0.96154 Mabsadd L
.74476 .86846 1.46946 -2.02254 Mabsadd L
.74476 .86846 .91448 -0.29713 Mabsadd L
.74476 .86846 2.37764 .77254 Mabsadd L
.74476 .86846 .56518 .7779 Mabsadd L
.74476 .86846 0 2.5 Mabsadd L closepath F
.74571 .86846 0 2.5 Mabsadd m
.74571 .86846 -0.56518 .7779 Mabsadd L
.74571 .86846 -2.37764 .77254 Mabsadd L
.74571 .86846 -0.91448 -0.29713 Mabsadd L
.74571 .86846 -1.46946 -2.02254 Mabsadd L
.74571 .86846 0 -0.96154 Mabsadd L
.74571 .86846 1.46946 -2.02254 Mabsadd L
.74571 .86846 .91448 -0.29713 Mabsadd L
.74571 .86846 2.37764 .77254 Mabsadd L
.74571 .86846 .56518 .7779 Mabsadd L
.74571 .86846 0 2.5 Mabsadd L closepath F
.74667 .86846 0 2.5 Mabsadd m
.74667 .86846 -0.56518 .7779 Mabsadd L
.74667 .86846 -2.37764 .77254 Mabsadd L
.74667 .86846 -0.91448 -0.29713 Mabsadd L
.74667 .86846 -1.46946 -2.02254 Mabsadd L
.74667 .86846 0 -0.96154 Mabsadd L
.74667 .86846 1.46946 -2.02254 Mabsadd L
.74667 .86846 .91448 -0.29713 Mabsadd L
.74667 .86846 2.37764 .77254 Mabsadd L
.74667 .86846 .56518 .7779 Mabsadd L
.74667 .86846 0 2.5 Mabsadd L closepath F
.74762 .86846 0 2.5 Mabsadd m
.74762 .86846 -0.56518 .7779 Mabsadd L
.74762 .86846 -2.37764 .77254 Mabsadd L
.74762 .86846 -0.91448 -0.29713 Mabsadd L
.74762 .86846 -1.46946 -2.02254 Mabsadd L
.74762 .86846 0 -0.96154 Mabsadd L
.74762 .86846 1.46946 -2.02254 Mabsadd L
.74762 .86846 .91448 -0.29713 Mabsadd L
.74762 .86846 2.37764 .77254 Mabsadd L
.74762 .86846 .56518 .7779 Mabsadd L
.74762 .86846 0 2.5 Mabsadd L closepath F
.74857 .86845 0 2.5 Mabsadd m
.74857 .86845 -0.56518 .7779 Mabsadd L
.74857 .86845 -2.37764 .77254 Mabsadd L
.74857 .86845 -0.91448 -0.29713 Mabsadd L
.74857 .86845 -1.46946 -2.02254 Mabsadd L
.74857 .86845 0 -0.96154 Mabsadd L
.74857 .86845 1.46946 -2.02254 Mabsadd L
.74857 .86845 .91448 -0.29713 Mabsadd L
.74857 .86845 2.37764 .77254 Mabsadd L
.74857 .86845 .56518 .7779 Mabsadd L
.74857 .86845 0 2.5 Mabsadd L closepath F
.74952 .86844 0 2.5 Mabsadd m
.74952 .86844 -0.56518 .7779 Mabsadd L
.74952 .86844 -2.37764 .77254 Mabsadd L
.74952 .86844 -0.91448 -0.29713 Mabsadd L
.74952 .86844 -1.46946 -2.02254 Mabsadd L
.74952 .86844 0 -0.96154 Mabsadd L
.74952 .86844 1.46946 -2.02254 Mabsadd L
.74952 .86844 .91448 -0.29713 Mabsadd L
.74952 .86844 2.37764 .77254 Mabsadd L
.74952 .86844 .56518 .7779 Mabsadd L
.74952 .86844 0 2.5 Mabsadd L closepath F
.75048 .86844 0 2.5 Mabsadd m
.75048 .86844 -0.56518 .7779 Mabsadd L
.75048 .86844 -2.37764 .77254 Mabsadd L
.75048 .86844 -0.91448 -0.29713 Mabsadd L
.75048 .86844 -1.46946 -2.02254 Mabsadd L
.75048 .86844 0 -0.96154 Mabsadd L
.75048 .86844 1.46946 -2.02254 Mabsadd L
.75048 .86844 .91448 -0.29713 Mabsadd L
.75048 .86844 2.37764 .77254 Mabsadd L
.75048 .86844 .56518 .7779 Mabsadd L
.75048 .86844 0 2.5 Mabsadd L closepath F
.75143 .86844 0 2.5 Mabsadd m
.75143 .86844 -0.56518 .7779 Mabsadd L
.75143 .86844 -2.37764 .77254 Mabsadd L
.75143 .86844 -0.91448 -0.29713 Mabsadd L
.75143 .86844 -1.46946 -2.02254 Mabsadd L
.75143 .86844 0 -0.96154 Mabsadd L
.75143 .86844 1.46946 -2.02254 Mabsadd L
.75143 .86844 .91448 -0.29713 Mabsadd L
.75143 .86844 2.37764 .77254 Mabsadd L
.75143 .86844 .56518 .7779 Mabsadd L
.75143 .86844 0 2.5 Mabsadd L closepath F
.75238 .86844 0 2.5 Mabsadd m
.75238 .86844 -0.56518 .7779 Mabsadd L
.75238 .86844 -2.37764 .77254 Mabsadd L
.75238 .86844 -0.91448 -0.29713 Mabsadd L
.75238 .86844 -1.46946 -2.02254 Mabsadd L
.75238 .86844 0 -0.96154 Mabsadd L
.75238 .86844 1.46946 -2.02254 Mabsadd L
.75238 .86844 .91448 -0.29713 Mabsadd L
.75238 .86844 2.37764 .77254 Mabsadd L
.75238 .86844 .56518 .7779 Mabsadd L
.75238 .86844 0 2.5 Mabsadd L closepath F
.75333 .86843 0 2.5 Mabsadd m
.75333 .86843 -0.56518 .7779 Mabsadd L
.75333 .86843 -2.37764 .77254 Mabsadd L
.75333 .86843 -0.91448 -0.29713 Mabsadd L
.75333 .86843 -1.46946 -2.02254 Mabsadd L
.75333 .86843 0 -0.96154 Mabsadd L
.75333 .86843 1.46946 -2.02254 Mabsadd L
.75333 .86843 .91448 -0.29713 Mabsadd L
.75333 .86843 2.37764 .77254 Mabsadd L
.75333 .86843 .56518 .7779 Mabsadd L
.75333 .86843 0 2.5 Mabsadd L closepath F
.75429 .86843 0 2.5 Mabsadd m
.75429 .86843 -0.56518 .7779 Mabsadd L
.75429 .86843 -2.37764 .77254 Mabsadd L
.75429 .86843 -0.91448 -0.29713 Mabsadd L
.75429 .86843 -1.46946 -2.02254 Mabsadd L
.75429 .86843 0 -0.96154 Mabsadd L
.75429 .86843 1.46946 -2.02254 Mabsadd L
.75429 .86843 .91448 -0.29713 Mabsadd L
.75429 .86843 2.37764 .77254 Mabsadd L
.75429 .86843 .56518 .7779 Mabsadd L
.75429 .86843 0 2.5 Mabsadd L closepath F
.75524 .86842 0 2.5 Mabsadd m
.75524 .86842 -0.56518 .7779 Mabsadd L
.75524 .86842 -2.37764 .77254 Mabsadd L
.75524 .86842 -0.91448 -0.29713 Mabsadd L
.75524 .86842 -1.46946 -2.02254 Mabsadd L
.75524 .86842 0 -0.96154 Mabsadd L
.75524 .86842 1.46946 -2.02254 Mabsadd L
.75524 .86842 .91448 -0.29713 Mabsadd L
.75524 .86842 2.37764 .77254 Mabsadd L
.75524 .86842 .56518 .7779 Mabsadd L
.75524 .86842 0 2.5 Mabsadd L closepath F
.75619 .86839 0 2.5 Mabsadd m
.75619 .86839 -0.56518 .7779 Mabsadd L
.75619 .86839 -2.37764 .77254 Mabsadd L
.75619 .86839 -0.91448 -0.29713 Mabsadd L
.75619 .86839 -1.46946 -2.02254 Mabsadd L
.75619 .86839 0 -0.96154 Mabsadd L
.75619 .86839 1.46946 -2.02254 Mabsadd L
.75619 .86839 .91448 -0.29713 Mabsadd L
.75619 .86839 2.37764 .77254 Mabsadd L
.75619 .86839 .56518 .7779 Mabsadd L
.75619 .86839 0 2.5 Mabsadd L closepath F
.75714 .86838 0 2.5 Mabsadd m
.75714 .86838 -0.56518 .7779 Mabsadd L
.75714 .86838 -2.37764 .77254 Mabsadd L
.75714 .86838 -0.91448 -0.29713 Mabsadd L
.75714 .86838 -1.46946 -2.02254 Mabsadd L
.75714 .86838 0 -0.96154 Mabsadd L
.75714 .86838 1.46946 -2.02254 Mabsadd L
.75714 .86838 .91448 -0.29713 Mabsadd L
.75714 .86838 2.37764 .77254 Mabsadd L
.75714 .86838 .56518 .7779 Mabsadd L
.75714 .86838 0 2.5 Mabsadd L closepath F
.7581 .86834 0 2.5 Mabsadd m
.7581 .86834 -0.56518 .7779 Mabsadd L
.7581 .86834 -2.37764 .77254 Mabsadd L
.7581 .86834 -0.91448 -0.29713 Mabsadd L
.7581 .86834 -1.46946 -2.02254 Mabsadd L
.7581 .86834 0 -0.96154 Mabsadd L
.7581 .86834 1.46946 -2.02254 Mabsadd L
.7581 .86834 .91448 -0.29713 Mabsadd L
.7581 .86834 2.37764 .77254 Mabsadd L
.7581 .86834 .56518 .7779 Mabsadd L
.7581 .86834 0 2.5 Mabsadd L closepath F
.75905 .86833 0 2.5 Mabsadd m
.75905 .86833 -0.56518 .7779 Mabsadd L
.75905 .86833 -2.37764 .77254 Mabsadd L
.75905 .86833 -0.91448 -0.29713 Mabsadd L
.75905 .86833 -1.46946 -2.02254 Mabsadd L
.75905 .86833 0 -0.96154 Mabsadd L
.75905 .86833 1.46946 -2.02254 Mabsadd L
.75905 .86833 .91448 -0.29713 Mabsadd L
.75905 .86833 2.37764 .77254 Mabsadd L
.75905 .86833 .56518 .7779 Mabsadd L
.75905 .86833 0 2.5 Mabsadd L closepath F
.76 .86817 0 2.5 Mabsadd m
.76 .86817 -0.56518 .7779 Mabsadd L
.76 .86817 -2.37764 .77254 Mabsadd L
.76 .86817 -0.91448 -0.29713 Mabsadd L
.76 .86817 -1.46946 -2.02254 Mabsadd L
.76 .86817 0 -0.96154 Mabsadd L
.76 .86817 1.46946 -2.02254 Mabsadd L
.76 .86817 .91448 -0.29713 Mabsadd L
.76 .86817 2.37764 .77254 Mabsadd L
.76 .86817 .56518 .7779 Mabsadd L
.76 .86817 0 2.5 Mabsadd L closepath F
.76095 .8681 0 2.5 Mabsadd m
.76095 .8681 -0.56518 .7779 Mabsadd L
.76095 .8681 -2.37764 .77254 Mabsadd L
.76095 .8681 -0.91448 -0.29713 Mabsadd L
.76095 .8681 -1.46946 -2.02254 Mabsadd L
.76095 .8681 0 -0.96154 Mabsadd L
.76095 .8681 1.46946 -2.02254 Mabsadd L
.76095 .8681 .91448 -0.29713 Mabsadd L
.76095 .8681 2.37764 .77254 Mabsadd L
.76095 .8681 .56518 .7779 Mabsadd L
.76095 .8681 0 2.5 Mabsadd L closepath F
.7619 .86737 0 2.5 Mabsadd m
.7619 .86737 -0.56518 .7779 Mabsadd L
.7619 .86737 -2.37764 .77254 Mabsadd L
.7619 .86737 -0.91448 -0.29713 Mabsadd L
.7619 .86737 -1.46946 -2.02254 Mabsadd L
.7619 .86737 0 -0.96154 Mabsadd L
.7619 .86737 1.46946 -2.02254 Mabsadd L
.7619 .86737 .91448 -0.29713 Mabsadd L
.7619 .86737 2.37764 .77254 Mabsadd L
.7619 .86737 .56518 .7779 Mabsadd L
.7619 .86737 0 2.5 Mabsadd L closepath F
.76286 .86696 0 2.5 Mabsadd m
.76286 .86696 -0.56518 .7779 Mabsadd L
.76286 .86696 -2.37764 .77254 Mabsadd L
.76286 .86696 -0.91448 -0.29713 Mabsadd L
.76286 .86696 -1.46946 -2.02254 Mabsadd L
.76286 .86696 0 -0.96154 Mabsadd L
.76286 .86696 1.46946 -2.02254 Mabsadd L
.76286 .86696 .91448 -0.29713 Mabsadd L
.76286 .86696 2.37764 .77254 Mabsadd L
.76286 .86696 .56518 .7779 Mabsadd L
.76286 .86696 0 2.5 Mabsadd L closepath F
.76381 .86248 0 2.5 Mabsadd m
.76381 .86248 -0.56518 .7779 Mabsadd L
.76381 .86248 -2.37764 .77254 Mabsadd L
.76381 .86248 -0.91448 -0.29713 Mabsadd L
.76381 .86248 -1.46946 -2.02254 Mabsadd L
.76381 .86248 0 -0.96154 Mabsadd L
.76381 .86248 1.46946 -2.02254 Mabsadd L
.76381 .86248 .91448 -0.29713 Mabsadd L
.76381 .86248 2.37764 .77254 Mabsadd L
.76381 .86248 .56518 .7779 Mabsadd L
.76381 .86248 0 2.5 Mabsadd L closepath F
.76476 .85927 0 2.5 Mabsadd m
.76476 .85927 -0.56518 .7779 Mabsadd L
.76476 .85927 -2.37764 .77254 Mabsadd L
.76476 .85927 -0.91448 -0.29713 Mabsadd L
.76476 .85927 -1.46946 -2.02254 Mabsadd L
.76476 .85927 0 -0.96154 Mabsadd L
.76476 .85927 1.46946 -2.02254 Mabsadd L
.76476 .85927 .91448 -0.29713 Mabsadd L
.76476 .85927 2.37764 .77254 Mabsadd L
.76476 .85927 .56518 .7779 Mabsadd L
.76476 .85927 0 2.5 Mabsadd L closepath F
.76571 .84891 0 2.5 Mabsadd m
.76571 .84891 -0.56518 .7779 Mabsadd L
.76571 .84891 -2.37764 .77254 Mabsadd L
.76571 .84891 -0.91448 -0.29713 Mabsadd L
.76571 .84891 -1.46946 -2.02254 Mabsadd L
.76571 .84891 0 -0.96154 Mabsadd L
.76571 .84891 1.46946 -2.02254 Mabsadd L
.76571 .84891 .91448 -0.29713 Mabsadd L
.76571 .84891 2.37764 .77254 Mabsadd L
.76571 .84891 .56518 .7779 Mabsadd L
.76571 .84891 0 2.5 Mabsadd L closepath F
.76667 .81178 0 2.5 Mabsadd m
.76667 .81178 -0.56518 .7779 Mabsadd L
.76667 .81178 -2.37764 .77254 Mabsadd L
.76667 .81178 -0.91448 -0.29713 Mabsadd L
.76667 .81178 -1.46946 -2.02254 Mabsadd L
.76667 .81178 0 -0.96154 Mabsadd L
.76667 .81178 1.46946 -2.02254 Mabsadd L
.76667 .81178 .91448 -0.29713 Mabsadd L
.76667 .81178 2.37764 .77254 Mabsadd L
.76667 .81178 .56518 .7779 Mabsadd L
.76667 .81178 0 2.5 Mabsadd L closepath F
.76762 .8118 0 2.5 Mabsadd m
.76762 .8118 -0.56518 .7779 Mabsadd L
.76762 .8118 -2.37764 .77254 Mabsadd L
.76762 .8118 -0.91448 -0.29713 Mabsadd L
.76762 .8118 -1.46946 -2.02254 Mabsadd L
.76762 .8118 0 -0.96154 Mabsadd L
.76762 .8118 1.46946 -2.02254 Mabsadd L
.76762 .8118 .91448 -0.29713 Mabsadd L
.76762 .8118 2.37764 .77254 Mabsadd L
.76762 .8118 .56518 .7779 Mabsadd L
.76762 .8118 0 2.5 Mabsadd L closepath F
.76857 .81335 0 2.5 Mabsadd m
.76857 .81335 -0.56518 .7779 Mabsadd L
.76857 .81335 -2.37764 .77254 Mabsadd L
.76857 .81335 -0.91448 -0.29713 Mabsadd L
.76857 .81335 -1.46946 -2.02254 Mabsadd L
.76857 .81335 0 -0.96154 Mabsadd L
.76857 .81335 1.46946 -2.02254 Mabsadd L
.76857 .81335 .91448 -0.29713 Mabsadd L
.76857 .81335 2.37764 .77254 Mabsadd L
.76857 .81335 .56518 .7779 Mabsadd L
.76857 .81335 0 2.5 Mabsadd L closepath F
.76952 .84321 0 2.5 Mabsadd m
.76952 .84321 -0.56518 .7779 Mabsadd L
.76952 .84321 -2.37764 .77254 Mabsadd L
.76952 .84321 -0.91448 -0.29713 Mabsadd L
.76952 .84321 -1.46946 -2.02254 Mabsadd L
.76952 .84321 0 -0.96154 Mabsadd L
.76952 .84321 1.46946 -2.02254 Mabsadd L
.76952 .84321 .91448 -0.29713 Mabsadd L
.76952 .84321 2.37764 .77254 Mabsadd L
.76952 .84321 .56518 .7779 Mabsadd L
.76952 .84321 0 2.5 Mabsadd L closepath F
.77048 .88109 0 2.5 Mabsadd m
.77048 .88109 -0.56518 .7779 Mabsadd L
.77048 .88109 -2.37764 .77254 Mabsadd L
.77048 .88109 -0.91448 -0.29713 Mabsadd L
.77048 .88109 -1.46946 -2.02254 Mabsadd L
.77048 .88109 0 -0.96154 Mabsadd L
.77048 .88109 1.46946 -2.02254 Mabsadd L
.77048 .88109 .91448 -0.29713 Mabsadd L
.77048 .88109 2.37764 .77254 Mabsadd L
.77048 .88109 .56518 .7779 Mabsadd L
.77048 .88109 0 2.5 Mabsadd L closepath F
.77143 .88137 0 2.5 Mabsadd m
.77143 .88137 -0.56518 .7779 Mabsadd L
.77143 .88137 -2.37764 .77254 Mabsadd L
.77143 .88137 -0.91448 -0.29713 Mabsadd L
.77143 .88137 -1.46946 -2.02254 Mabsadd L
.77143 .88137 0 -0.96154 Mabsadd L
.77143 .88137 1.46946 -2.02254 Mabsadd L
.77143 .88137 .91448 -0.29713 Mabsadd L
.77143 .88137 2.37764 .77254 Mabsadd L
.77143 .88137 .56518 .7779 Mabsadd L
.77143 .88137 0 2.5 Mabsadd L closepath F
.77238 .88393 0 2.5 Mabsadd m
.77238 .88393 -0.56518 .7779 Mabsadd L
.77238 .88393 -2.37764 .77254 Mabsadd L
.77238 .88393 -0.91448 -0.29713 Mabsadd L
.77238 .88393 -1.46946 -2.02254 Mabsadd L
.77238 .88393 0 -0.96154 Mabsadd L
.77238 .88393 1.46946 -2.02254 Mabsadd L
.77238 .88393 .91448 -0.29713 Mabsadd L
.77238 .88393 2.37764 .77254 Mabsadd L
.77238 .88393 .56518 .7779 Mabsadd L
.77238 .88393 0 2.5 Mabsadd L closepath F
.77333 .8873 0 2.5 Mabsadd m
.77333 .8873 -0.56518 .7779 Mabsadd L
.77333 .8873 -2.37764 .77254 Mabsadd L
.77333 .8873 -0.91448 -0.29713 Mabsadd L
.77333 .8873 -1.46946 -2.02254 Mabsadd L
.77333 .8873 0 -0.96154 Mabsadd L
.77333 .8873 1.46946 -2.02254 Mabsadd L
.77333 .8873 .91448 -0.29713 Mabsadd L
.77333 .8873 2.37764 .77254 Mabsadd L
.77333 .8873 .56518 .7779 Mabsadd L
.77333 .8873 0 2.5 Mabsadd L closepath F
.77429 .88717 0 2.5 Mabsadd m
.77429 .88717 -0.56518 .7779 Mabsadd L
.77429 .88717 -2.37764 .77254 Mabsadd L
.77429 .88717 -0.91448 -0.29713 Mabsadd L
.77429 .88717 -1.46946 -2.02254 Mabsadd L
.77429 .88717 0 -0.96154 Mabsadd L
.77429 .88717 1.46946 -2.02254 Mabsadd L
.77429 .88717 .91448 -0.29713 Mabsadd L
.77429 .88717 2.37764 .77254 Mabsadd L
.77429 .88717 .56518 .7779 Mabsadd L
.77429 .88717 0 2.5 Mabsadd L closepath F
.77524 .87344 0 2.5 Mabsadd m
.77524 .87344 -0.56518 .7779 Mabsadd L
.77524 .87344 -2.37764 .77254 Mabsadd L
.77524 .87344 -0.91448 -0.29713 Mabsadd L
.77524 .87344 -1.46946 -2.02254 Mabsadd L
.77524 .87344 0 -0.96154 Mabsadd L
.77524 .87344 1.46946 -2.02254 Mabsadd L
.77524 .87344 .91448 -0.29713 Mabsadd L
.77524 .87344 2.37764 .77254 Mabsadd L
.77524 .87344 .56518 .7779 Mabsadd L
.77524 .87344 0 2.5 Mabsadd L closepath F
.77619 .88652 0 2.5 Mabsadd m
.77619 .88652 -0.56518 .7779 Mabsadd L
.77619 .88652 -2.37764 .77254 Mabsadd L
.77619 .88652 -0.91448 -0.29713 Mabsadd L
.77619 .88652 -1.46946 -2.02254 Mabsadd L
.77619 .88652 0 -0.96154 Mabsadd L
.77619 .88652 1.46946 -2.02254 Mabsadd L
.77619 .88652 .91448 -0.29713 Mabsadd L
.77619 .88652 2.37764 .77254 Mabsadd L
.77619 .88652 .56518 .7779 Mabsadd L
.77619 .88652 0 2.5 Mabsadd L closepath F
.77714 .88767 0 2.5 Mabsadd m
.77714 .88767 -0.56518 .7779 Mabsadd L
.77714 .88767 -2.37764 .77254 Mabsadd L
.77714 .88767 -0.91448 -0.29713 Mabsadd L
.77714 .88767 -1.46946 -2.02254 Mabsadd L
.77714 .88767 0 -0.96154 Mabsadd L
.77714 .88767 1.46946 -2.02254 Mabsadd L
.77714 .88767 .91448 -0.29713 Mabsadd L
.77714 .88767 2.37764 .77254 Mabsadd L
.77714 .88767 .56518 .7779 Mabsadd L
.77714 .88767 0 2.5 Mabsadd L closepath F
.7781 .89103 0 2.5 Mabsadd m
.7781 .89103 -0.56518 .7779 Mabsadd L
.7781 .89103 -2.37764 .77254 Mabsadd L
.7781 .89103 -0.91448 -0.29713 Mabsadd L
.7781 .89103 -1.46946 -2.02254 Mabsadd L
.7781 .89103 0 -0.96154 Mabsadd L
.7781 .89103 1.46946 -2.02254 Mabsadd L
.7781 .89103 .91448 -0.29713 Mabsadd L
.7781 .89103 2.37764 .77254 Mabsadd L
.7781 .89103 .56518 .7779 Mabsadd L
.7781 .89103 0 2.5 Mabsadd L closepath F
.77905 .8927 0 2.5 Mabsadd m
.77905 .8927 -0.56518 .7779 Mabsadd L
.77905 .8927 -2.37764 .77254 Mabsadd L
.77905 .8927 -0.91448 -0.29713 Mabsadd L
.77905 .8927 -1.46946 -2.02254 Mabsadd L
.77905 .8927 0 -0.96154 Mabsadd L
.77905 .8927 1.46946 -2.02254 Mabsadd L
.77905 .8927 .91448 -0.29713 Mabsadd L
.77905 .8927 2.37764 .77254 Mabsadd L
.77905 .8927 .56518 .7779 Mabsadd L
.77905 .8927 0 2.5 Mabsadd L closepath F
.78 .91038 0 2.5 Mabsadd m
.78 .91038 -0.56518 .7779 Mabsadd L
.78 .91038 -2.37764 .77254 Mabsadd L
.78 .91038 -0.91448 -0.29713 Mabsadd L
.78 .91038 -1.46946 -2.02254 Mabsadd L
.78 .91038 0 -0.96154 Mabsadd L
.78 .91038 1.46946 -2.02254 Mabsadd L
.78 .91038 .91448 -0.29713 Mabsadd L
.78 .91038 2.37764 .77254 Mabsadd L
.78 .91038 .56518 .7779 Mabsadd L
.78 .91038 0 2.5 Mabsadd L closepath F
.78095 .91642 0 2.5 Mabsadd m
.78095 .91642 -0.56518 .7779 Mabsadd L
.78095 .91642 -2.37764 .77254 Mabsadd L
.78095 .91642 -0.91448 -0.29713 Mabsadd L
.78095 .91642 -1.46946 -2.02254 Mabsadd L
.78095 .91642 0 -0.96154 Mabsadd L
.78095 .91642 1.46946 -2.02254 Mabsadd L
.78095 .91642 .91448 -0.29713 Mabsadd L
.78095 .91642 2.37764 .77254 Mabsadd L
.78095 .91642 .56518 .7779 Mabsadd L
.78095 .91642 0 2.5 Mabsadd L closepath F
.7819 .91846 0 2.5 Mabsadd m
.7819 .91846 -0.56518 .7779 Mabsadd L
.7819 .91846 -2.37764 .77254 Mabsadd L
.7819 .91846 -0.91448 -0.29713 Mabsadd L
.7819 .91846 -1.46946 -2.02254 Mabsadd L
.7819 .91846 0 -0.96154 Mabsadd L
.7819 .91846 1.46946 -2.02254 Mabsadd L
.7819 .91846 .91448 -0.29713 Mabsadd L
.7819 .91846 2.37764 .77254 Mabsadd L
.7819 .91846 .56518 .7779 Mabsadd L
.7819 .91846 0 2.5 Mabsadd L closepath F
.78286 .92445 0 2.5 Mabsadd m
.78286 .92445 -0.56518 .7779 Mabsadd L
.78286 .92445 -2.37764 .77254 Mabsadd L
.78286 .92445 -0.91448 -0.29713 Mabsadd L
.78286 .92445 -1.46946 -2.02254 Mabsadd L
.78286 .92445 0 -0.96154 Mabsadd L
.78286 .92445 1.46946 -2.02254 Mabsadd L
.78286 .92445 .91448 -0.29713 Mabsadd L
.78286 .92445 2.37764 .77254 Mabsadd L
.78286 .92445 .56518 .7779 Mabsadd L
.78286 .92445 0 2.5 Mabsadd L closepath F
.78381 .92475 0 2.5 Mabsadd m
.78381 .92475 -0.56518 .7779 Mabsadd L
.78381 .92475 -2.37764 .77254 Mabsadd L
.78381 .92475 -0.91448 -0.29713 Mabsadd L
.78381 .92475 -1.46946 -2.02254 Mabsadd L
.78381 .92475 0 -0.96154 Mabsadd L
.78381 .92475 1.46946 -2.02254 Mabsadd L
.78381 .92475 .91448 -0.29713 Mabsadd L
.78381 .92475 2.37764 .77254 Mabsadd L
.78381 .92475 .56518 .7779 Mabsadd L
.78381 .92475 0 2.5 Mabsadd L closepath F
.78476 .92569 0 2.5 Mabsadd m
.78476 .92569 -0.56518 .7779 Mabsadd L
.78476 .92569 -2.37764 .77254 Mabsadd L
.78476 .92569 -0.91448 -0.29713 Mabsadd L
.78476 .92569 -1.46946 -2.02254 Mabsadd L
.78476 .92569 0 -0.96154 Mabsadd L
.78476 .92569 1.46946 -2.02254 Mabsadd L
.78476 .92569 .91448 -0.29713 Mabsadd L
.78476 .92569 2.37764 .77254 Mabsadd L
.78476 .92569 .56518 .7779 Mabsadd L
.78476 .92569 0 2.5 Mabsadd L closepath F
.78571 .92575 0 2.5 Mabsadd m
.78571 .92575 -0.56518 .7779 Mabsadd L
.78571 .92575 -2.37764 .77254 Mabsadd L
.78571 .92575 -0.91448 -0.29713 Mabsadd L
.78571 .92575 -1.46946 -2.02254 Mabsadd L
.78571 .92575 0 -0.96154 Mabsadd L
.78571 .92575 1.46946 -2.02254 Mabsadd L
.78571 .92575 .91448 -0.29713 Mabsadd L
.78571 .92575 2.37764 .77254 Mabsadd L
.78571 .92575 .56518 .7779 Mabsadd L
.78571 .92575 0 2.5 Mabsadd L closepath F
.78667 .92594 0 2.5 Mabsadd m
.78667 .92594 -0.56518 .7779 Mabsadd L
.78667 .92594 -2.37764 .77254 Mabsadd L
.78667 .92594 -0.91448 -0.29713 Mabsadd L
.78667 .92594 -1.46946 -2.02254 Mabsadd L
.78667 .92594 0 -0.96154 Mabsadd L
.78667 .92594 1.46946 -2.02254 Mabsadd L
.78667 .92594 .91448 -0.29713 Mabsadd L
.78667 .92594 2.37764 .77254 Mabsadd L
.78667 .92594 .56518 .7779 Mabsadd L
.78667 .92594 0 2.5 Mabsadd L closepath F
.78762 .92595 0 2.5 Mabsadd m
.78762 .92595 -0.56518 .7779 Mabsadd L
.78762 .92595 -2.37764 .77254 Mabsadd L
.78762 .92595 -0.91448 -0.29713 Mabsadd L
.78762 .92595 -1.46946 -2.02254 Mabsadd L
.78762 .92595 0 -0.96154 Mabsadd L
.78762 .92595 1.46946 -2.02254 Mabsadd L
.78762 .92595 .91448 -0.29713 Mabsadd L
.78762 .92595 2.37764 .77254 Mabsadd L
.78762 .92595 .56518 .7779 Mabsadd L
.78762 .92595 0 2.5 Mabsadd L closepath F
.78857 .92601 0 2.5 Mabsadd m
.78857 .92601 -0.56518 .7779 Mabsadd L
.78857 .92601 -2.37764 .77254 Mabsadd L
.78857 .92601 -0.91448 -0.29713 Mabsadd L
.78857 .92601 -1.46946 -2.02254 Mabsadd L
.78857 .92601 0 -0.96154 Mabsadd L
.78857 .92601 1.46946 -2.02254 Mabsadd L
.78857 .92601 .91448 -0.29713 Mabsadd L
.78857 .92601 2.37764 .77254 Mabsadd L
.78857 .92601 .56518 .7779 Mabsadd L
.78857 .92601 0 2.5 Mabsadd L closepath F
.78952 .92609 0 2.5 Mabsadd m
.78952 .92609 -0.56518 .7779 Mabsadd L
.78952 .92609 -2.37764 .77254 Mabsadd L
.78952 .92609 -0.91448 -0.29713 Mabsadd L
.78952 .92609 -1.46946 -2.02254 Mabsadd L
.78952 .92609 0 -0.96154 Mabsadd L
.78952 .92609 1.46946 -2.02254 Mabsadd L
.78952 .92609 .91448 -0.29713 Mabsadd L
.78952 .92609 2.37764 .77254 Mabsadd L
.78952 .92609 .56518 .7779 Mabsadd L
.78952 .92609 0 2.5 Mabsadd L closepath F
.79048 .92609 0 2.5 Mabsadd m
.79048 .92609 -0.56518 .7779 Mabsadd L
.79048 .92609 -2.37764 .77254 Mabsadd L
.79048 .92609 -0.91448 -0.29713 Mabsadd L
.79048 .92609 -1.46946 -2.02254 Mabsadd L
.79048 .92609 0 -0.96154 Mabsadd L
.79048 .92609 1.46946 -2.02254 Mabsadd L
.79048 .92609 .91448 -0.29713 Mabsadd L
.79048 .92609 2.37764 .77254 Mabsadd L
.79048 .92609 .56518 .7779 Mabsadd L
.79048 .92609 0 2.5 Mabsadd L closepath F
.79143 .9261 0 2.5 Mabsadd m
.79143 .9261 -0.56518 .7779 Mabsadd L
.79143 .9261 -2.37764 .77254 Mabsadd L
.79143 .9261 -0.91448 -0.29713 Mabsadd L
.79143 .9261 -1.46946 -2.02254 Mabsadd L
.79143 .9261 0 -0.96154 Mabsadd L
.79143 .9261 1.46946 -2.02254 Mabsadd L
.79143 .9261 .91448 -0.29713 Mabsadd L
.79143 .9261 2.37764 .77254 Mabsadd L
.79143 .9261 .56518 .7779 Mabsadd L
.79143 .9261 0 2.5 Mabsadd L closepath F
.79238 .92611 0 2.5 Mabsadd m
.79238 .92611 -0.56518 .7779 Mabsadd L
.79238 .92611 -2.37764 .77254 Mabsadd L
.79238 .92611 -0.91448 -0.29713 Mabsadd L
.79238 .92611 -1.46946 -2.02254 Mabsadd L
.79238 .92611 0 -0.96154 Mabsadd L
.79238 .92611 1.46946 -2.02254 Mabsadd L
.79238 .92611 .91448 -0.29713 Mabsadd L
.79238 .92611 2.37764 .77254 Mabsadd L
.79238 .92611 .56518 .7779 Mabsadd L
.79238 .92611 0 2.5 Mabsadd L closepath F
.79333 .92611 0 2.5 Mabsadd m
.79333 .92611 -0.56518 .7779 Mabsadd L
.79333 .92611 -2.37764 .77254 Mabsadd L
.79333 .92611 -0.91448 -0.29713 Mabsadd L
.79333 .92611 -1.46946 -2.02254 Mabsadd L
.79333 .92611 0 -0.96154 Mabsadd L
.79333 .92611 1.46946 -2.02254 Mabsadd L
.79333 .92611 .91448 -0.29713 Mabsadd L
.79333 .92611 2.37764 .77254 Mabsadd L
.79333 .92611 .56518 .7779 Mabsadd L
.79333 .92611 0 2.5 Mabsadd L closepath F
.79429 .92612 0 2.5 Mabsadd m
.79429 .92612 -0.56518 .7779 Mabsadd L
.79429 .92612 -2.37764 .77254 Mabsadd L
.79429 .92612 -0.91448 -0.29713 Mabsadd L
.79429 .92612 -1.46946 -2.02254 Mabsadd L
.79429 .92612 0 -0.96154 Mabsadd L
.79429 .92612 1.46946 -2.02254 Mabsadd L
.79429 .92612 .91448 -0.29713 Mabsadd L
.79429 .92612 2.37764 .77254 Mabsadd L
.79429 .92612 .56518 .7779 Mabsadd L
.79429 .92612 0 2.5 Mabsadd L closepath F
.79524 .92612 0 2.5 Mabsadd m
.79524 .92612 -0.56518 .7779 Mabsadd L
.79524 .92612 -2.37764 .77254 Mabsadd L
.79524 .92612 -0.91448 -0.29713 Mabsadd L
.79524 .92612 -1.46946 -2.02254 Mabsadd L
.79524 .92612 0 -0.96154 Mabsadd L
.79524 .92612 1.46946 -2.02254 Mabsadd L
.79524 .92612 .91448 -0.29713 Mabsadd L
.79524 .92612 2.37764 .77254 Mabsadd L
.79524 .92612 .56518 .7779 Mabsadd L
.79524 .92612 0 2.5 Mabsadd L closepath F
.79619 .92613 0 2.5 Mabsadd m
.79619 .92613 -0.56518 .7779 Mabsadd L
.79619 .92613 -2.37764 .77254 Mabsadd L
.79619 .92613 -0.91448 -0.29713 Mabsadd L
.79619 .92613 -1.46946 -2.02254 Mabsadd L
.79619 .92613 0 -0.96154 Mabsadd L
.79619 .92613 1.46946 -2.02254 Mabsadd L
.79619 .92613 .91448 -0.29713 Mabsadd L
.79619 .92613 2.37764 .77254 Mabsadd L
.79619 .92613 .56518 .7779 Mabsadd L
.79619 .92613 0 2.5 Mabsadd L closepath F
.79714 .92613 0 2.5 Mabsadd m
.79714 .92613 -0.56518 .7779 Mabsadd L
.79714 .92613 -2.37764 .77254 Mabsadd L
.79714 .92613 -0.91448 -0.29713 Mabsadd L
.79714 .92613 -1.46946 -2.02254 Mabsadd L
.79714 .92613 0 -0.96154 Mabsadd L
.79714 .92613 1.46946 -2.02254 Mabsadd L
.79714 .92613 .91448 -0.29713 Mabsadd L
.79714 .92613 2.37764 .77254 Mabsadd L
.79714 .92613 .56518 .7779 Mabsadd L
.79714 .92613 0 2.5 Mabsadd L closepath F
.7981 .92613 0 2.5 Mabsadd m
.7981 .92613 -0.56518 .7779 Mabsadd L
.7981 .92613 -2.37764 .77254 Mabsadd L
.7981 .92613 -0.91448 -0.29713 Mabsadd L
.7981 .92613 -1.46946 -2.02254 Mabsadd L
.7981 .92613 0 -0.96154 Mabsadd L
.7981 .92613 1.46946 -2.02254 Mabsadd L
.7981 .92613 .91448 -0.29713 Mabsadd L
.7981 .92613 2.37764 .77254 Mabsadd L
.7981 .92613 .56518 .7779 Mabsadd L
.7981 .92613 0 2.5 Mabsadd L closepath F
.79905 .92613 0 2.5 Mabsadd m
.79905 .92613 -0.56518 .7779 Mabsadd L
.79905 .92613 -2.37764 .77254 Mabsadd L
.79905 .92613 -0.91448 -0.29713 Mabsadd L
.79905 .92613 -1.46946 -2.02254 Mabsadd L
.79905 .92613 0 -0.96154 Mabsadd L
.79905 .92613 1.46946 -2.02254 Mabsadd L
.79905 .92613 .91448 -0.29713 Mabsadd L
.79905 .92613 2.37764 .77254 Mabsadd L
.79905 .92613 .56518 .7779 Mabsadd L
.79905 .92613 0 2.5 Mabsadd L closepath F
.8 .92613 0 2.5 Mabsadd m
.8 .92613 -0.56518 .7779 Mabsadd L
.8 .92613 -2.37764 .77254 Mabsadd L
.8 .92613 -0.91448 -0.29713 Mabsadd L
.8 .92613 -1.46946 -2.02254 Mabsadd L
.8 .92613 0 -0.96154 Mabsadd L
.8 .92613 1.46946 -2.02254 Mabsadd L
.8 .92613 .91448 -0.29713 Mabsadd L
.8 .92613 2.37764 .77254 Mabsadd L
.8 .92613 .56518 .7779 Mabsadd L
.8 .92613 0 2.5 Mabsadd L closepath F
.80095 .92613 0 2.5 Mabsadd m
.80095 .92613 -0.56518 .7779 Mabsadd L
.80095 .92613 -2.37764 .77254 Mabsadd L
.80095 .92613 -0.91448 -0.29713 Mabsadd L
.80095 .92613 -1.46946 -2.02254 Mabsadd L
.80095 .92613 0 -0.96154 Mabsadd L
.80095 .92613 1.46946 -2.02254 Mabsadd L
.80095 .92613 .91448 -0.29713 Mabsadd L
.80095 .92613 2.37764 .77254 Mabsadd L
.80095 .92613 .56518 .7779 Mabsadd L
.80095 .92613 0 2.5 Mabsadd L closepath F
.8019 .92612 0 2.5 Mabsadd m
.8019 .92612 -0.56518 .7779 Mabsadd L
.8019 .92612 -2.37764 .77254 Mabsadd L
.8019 .92612 -0.91448 -0.29713 Mabsadd L
.8019 .92612 -1.46946 -2.02254 Mabsadd L
.8019 .92612 0 -0.96154 Mabsadd L
.8019 .92612 1.46946 -2.02254 Mabsadd L
.8019 .92612 .91448 -0.29713 Mabsadd L
.8019 .92612 2.37764 .77254 Mabsadd L
.8019 .92612 .56518 .7779 Mabsadd L
.8019 .92612 0 2.5 Mabsadd L closepath F
.80286 .92612 0 2.5 Mabsadd m
.80286 .92612 -0.56518 .7779 Mabsadd L
.80286 .92612 -2.37764 .77254 Mabsadd L
.80286 .92612 -0.91448 -0.29713 Mabsadd L
.80286 .92612 -1.46946 -2.02254 Mabsadd L
.80286 .92612 0 -0.96154 Mabsadd L
.80286 .92612 1.46946 -2.02254 Mabsadd L
.80286 .92612 .91448 -0.29713 Mabsadd L
.80286 .92612 2.37764 .77254 Mabsadd L
.80286 .92612 .56518 .7779 Mabsadd L
.80286 .92612 0 2.5 Mabsadd L closepath F
.80381 .92612 0 2.5 Mabsadd m
.80381 .92612 -0.56518 .7779 Mabsadd L
.80381 .92612 -2.37764 .77254 Mabsadd L
.80381 .92612 -0.91448 -0.29713 Mabsadd L
.80381 .92612 -1.46946 -2.02254 Mabsadd L
.80381 .92612 0 -0.96154 Mabsadd L
.80381 .92612 1.46946 -2.02254 Mabsadd L
.80381 .92612 .91448 -0.29713 Mabsadd L
.80381 .92612 2.37764 .77254 Mabsadd L
.80381 .92612 .56518 .7779 Mabsadd L
.80381 .92612 0 2.5 Mabsadd L closepath F
.80476 .92612 0 2.5 Mabsadd m
.80476 .92612 -0.56518 .7779 Mabsadd L
.80476 .92612 -2.37764 .77254 Mabsadd L
.80476 .92612 -0.91448 -0.29713 Mabsadd L
.80476 .92612 -1.46946 -2.02254 Mabsadd L
.80476 .92612 0 -0.96154 Mabsadd L
.80476 .92612 1.46946 -2.02254 Mabsadd L
.80476 .92612 .91448 -0.29713 Mabsadd L
.80476 .92612 2.37764 .77254 Mabsadd L
.80476 .92612 .56518 .7779 Mabsadd L
.80476 .92612 0 2.5 Mabsadd L closepath F
.80571 .92611 0 2.5 Mabsadd m
.80571 .92611 -0.56518 .7779 Mabsadd L
.80571 .92611 -2.37764 .77254 Mabsadd L
.80571 .92611 -0.91448 -0.29713 Mabsadd L
.80571 .92611 -1.46946 -2.02254 Mabsadd L
.80571 .92611 0 -0.96154 Mabsadd L
.80571 .92611 1.46946 -2.02254 Mabsadd L
.80571 .92611 .91448 -0.29713 Mabsadd L
.80571 .92611 2.37764 .77254 Mabsadd L
.80571 .92611 .56518 .7779 Mabsadd L
.80571 .92611 0 2.5 Mabsadd L closepath F
.80667 .9261 0 2.5 Mabsadd m
.80667 .9261 -0.56518 .7779 Mabsadd L
.80667 .9261 -2.37764 .77254 Mabsadd L
.80667 .9261 -0.91448 -0.29713 Mabsadd L
.80667 .9261 -1.46946 -2.02254 Mabsadd L
.80667 .9261 0 -0.96154 Mabsadd L
.80667 .9261 1.46946 -2.02254 Mabsadd L
.80667 .9261 .91448 -0.29713 Mabsadd L
.80667 .9261 2.37764 .77254 Mabsadd L
.80667 .9261 .56518 .7779 Mabsadd L
.80667 .9261 0 2.5 Mabsadd L closepath F
.80762 .9261 0 2.5 Mabsadd m
.80762 .9261 -0.56518 .7779 Mabsadd L
.80762 .9261 -2.37764 .77254 Mabsadd L
.80762 .9261 -0.91448 -0.29713 Mabsadd L
.80762 .9261 -1.46946 -2.02254 Mabsadd L
.80762 .9261 0 -0.96154 Mabsadd L
.80762 .9261 1.46946 -2.02254 Mabsadd L
.80762 .9261 .91448 -0.29713 Mabsadd L
.80762 .9261 2.37764 .77254 Mabsadd L
.80762 .9261 .56518 .7779 Mabsadd L
.80762 .9261 0 2.5 Mabsadd L closepath F
.80857 .92609 0 2.5 Mabsadd m
.80857 .92609 -0.56518 .7779 Mabsadd L
.80857 .92609 -2.37764 .77254 Mabsadd L
.80857 .92609 -0.91448 -0.29713 Mabsadd L
.80857 .92609 -1.46946 -2.02254 Mabsadd L
.80857 .92609 0 -0.96154 Mabsadd L
.80857 .92609 1.46946 -2.02254 Mabsadd L
.80857 .92609 .91448 -0.29713 Mabsadd L
.80857 .92609 2.37764 .77254 Mabsadd L
.80857 .92609 .56518 .7779 Mabsadd L
.80857 .92609 0 2.5 Mabsadd L closepath F
.80952 .92606 0 2.5 Mabsadd m
.80952 .92606 -0.56518 .7779 Mabsadd L
.80952 .92606 -2.37764 .77254 Mabsadd L
.80952 .92606 -0.91448 -0.29713 Mabsadd L
.80952 .92606 -1.46946 -2.02254 Mabsadd L
.80952 .92606 0 -0.96154 Mabsadd L
.80952 .92606 1.46946 -2.02254 Mabsadd L
.80952 .92606 .91448 -0.29713 Mabsadd L
.80952 .92606 2.37764 .77254 Mabsadd L
.80952 .92606 .56518 .7779 Mabsadd L
.80952 .92606 0 2.5 Mabsadd L closepath F
.81048 .92606 0 2.5 Mabsadd m
.81048 .92606 -0.56518 .7779 Mabsadd L
.81048 .92606 -2.37764 .77254 Mabsadd L
.81048 .92606 -0.91448 -0.29713 Mabsadd L
.81048 .92606 -1.46946 -2.02254 Mabsadd L
.81048 .92606 0 -0.96154 Mabsadd L
.81048 .92606 1.46946 -2.02254 Mabsadd L
.81048 .92606 .91448 -0.29713 Mabsadd L
.81048 .92606 2.37764 .77254 Mabsadd L
.81048 .92606 .56518 .7779 Mabsadd L
.81048 .92606 0 2.5 Mabsadd L closepath F
.81143 .92602 0 2.5 Mabsadd m
.81143 .92602 -0.56518 .7779 Mabsadd L
.81143 .92602 -2.37764 .77254 Mabsadd L
.81143 .92602 -0.91448 -0.29713 Mabsadd L
.81143 .92602 -1.46946 -2.02254 Mabsadd L
.81143 .92602 0 -0.96154 Mabsadd L
.81143 .92602 1.46946 -2.02254 Mabsadd L
.81143 .92602 .91448 -0.29713 Mabsadd L
.81143 .92602 2.37764 .77254 Mabsadd L
.81143 .92602 .56518 .7779 Mabsadd L
.81143 .92602 0 2.5 Mabsadd L closepath F
.81238 .92601 0 2.5 Mabsadd m
.81238 .92601 -0.56518 .7779 Mabsadd L
.81238 .92601 -2.37764 .77254 Mabsadd L
.81238 .92601 -0.91448 -0.29713 Mabsadd L
.81238 .92601 -1.46946 -2.02254 Mabsadd L
.81238 .92601 0 -0.96154 Mabsadd L
.81238 .92601 1.46946 -2.02254 Mabsadd L
.81238 .92601 .91448 -0.29713 Mabsadd L
.81238 .92601 2.37764 .77254 Mabsadd L
.81238 .92601 .56518 .7779 Mabsadd L
.81238 .92601 0 2.5 Mabsadd L closepath F
.81333 .92587 0 2.5 Mabsadd m
.81333 .92587 -0.56518 .7779 Mabsadd L
.81333 .92587 -2.37764 .77254 Mabsadd L
.81333 .92587 -0.91448 -0.29713 Mabsadd L
.81333 .92587 -1.46946 -2.02254 Mabsadd L
.81333 .92587 0 -0.96154 Mabsadd L
.81333 .92587 1.46946 -2.02254 Mabsadd L
.81333 .92587 .91448 -0.29713 Mabsadd L
.81333 .92587 2.37764 .77254 Mabsadd L
.81333 .92587 .56518 .7779 Mabsadd L
.81333 .92587 0 2.5 Mabsadd L closepath F
.81429 .9258 0 2.5 Mabsadd m
.81429 .9258 -0.56518 .7779 Mabsadd L
.81429 .9258 -2.37764 .77254 Mabsadd L
.81429 .9258 -0.91448 -0.29713 Mabsadd L
.81429 .9258 -1.46946 -2.02254 Mabsadd L
.81429 .9258 0 -0.96154 Mabsadd L
.81429 .9258 1.46946 -2.02254 Mabsadd L
.81429 .9258 .91448 -0.29713 Mabsadd L
.81429 .9258 2.37764 .77254 Mabsadd L
.81429 .9258 .56518 .7779 Mabsadd L
.81429 .9258 0 2.5 Mabsadd L closepath F
.81524 .92515 0 2.5 Mabsadd m
.81524 .92515 -0.56518 .7779 Mabsadd L
.81524 .92515 -2.37764 .77254 Mabsadd L
.81524 .92515 -0.91448 -0.29713 Mabsadd L
.81524 .92515 -1.46946 -2.02254 Mabsadd L
.81524 .92515 0 -0.96154 Mabsadd L
.81524 .92515 1.46946 -2.02254 Mabsadd L
.81524 .92515 .91448 -0.29713 Mabsadd L
.81524 .92515 2.37764 .77254 Mabsadd L
.81524 .92515 .56518 .7779 Mabsadd L
.81524 .92515 0 2.5 Mabsadd L closepath F
.81619 .92479 0 2.5 Mabsadd m
.81619 .92479 -0.56518 .7779 Mabsadd L
.81619 .92479 -2.37764 .77254 Mabsadd L
.81619 .92479 -0.91448 -0.29713 Mabsadd L
.81619 .92479 -1.46946 -2.02254 Mabsadd L
.81619 .92479 0 -0.96154 Mabsadd L
.81619 .92479 1.46946 -2.02254 Mabsadd L
.81619 .92479 .91448 -0.29713 Mabsadd L
.81619 .92479 2.37764 .77254 Mabsadd L
.81619 .92479 .56518 .7779 Mabsadd L
.81619 .92479 0 2.5 Mabsadd L closepath F
.81714 .92084 0 2.5 Mabsadd m
.81714 .92084 -0.56518 .7779 Mabsadd L
.81714 .92084 -2.37764 .77254 Mabsadd L
.81714 .92084 -0.91448 -0.29713 Mabsadd L
.81714 .92084 -1.46946 -2.02254 Mabsadd L
.81714 .92084 0 -0.96154 Mabsadd L
.81714 .92084 1.46946 -2.02254 Mabsadd L
.81714 .92084 .91448 -0.29713 Mabsadd L
.81714 .92084 2.37764 .77254 Mabsadd L
.81714 .92084 .56518 .7779 Mabsadd L
.81714 .92084 0 2.5 Mabsadd L closepath F
.8181 .91806 0 2.5 Mabsadd m
.8181 .91806 -0.56518 .7779 Mabsadd L
.8181 .91806 -2.37764 .77254 Mabsadd L
.8181 .91806 -0.91448 -0.29713 Mabsadd L
.8181 .91806 -1.46946 -2.02254 Mabsadd L
.8181 .91806 0 -0.96154 Mabsadd L
.8181 .91806 1.46946 -2.02254 Mabsadd L
.8181 .91806 .91448 -0.29713 Mabsadd L
.8181 .91806 2.37764 .77254 Mabsadd L
.8181 .91806 .56518 .7779 Mabsadd L
.8181 .91806 0 2.5 Mabsadd L closepath F
.81905 .90921 0 2.5 Mabsadd m
.81905 .90921 -0.56518 .7779 Mabsadd L
.81905 .90921 -2.37764 .77254 Mabsadd L
.81905 .90921 -0.91448 -0.29713 Mabsadd L
.81905 .90921 -1.46946 -2.02254 Mabsadd L
.81905 .90921 0 -0.96154 Mabsadd L
.81905 .90921 1.46946 -2.02254 Mabsadd L
.81905 .90921 .91448 -0.29713 Mabsadd L
.81905 .90921 2.37764 .77254 Mabsadd L
.81905 .90921 .56518 .7779 Mabsadd L
.81905 .90921 0 2.5 Mabsadd L closepath F
.82 .87792 0 2.5 Mabsadd m
.82 .87792 -0.56518 .7779 Mabsadd L
.82 .87792 -2.37764 .77254 Mabsadd L
.82 .87792 -0.91448 -0.29713 Mabsadd L
.82 .87792 -1.46946 -2.02254 Mabsadd L
.82 .87792 0 -0.96154 Mabsadd L
.82 .87792 1.46946 -2.02254 Mabsadd L
.82 .87792 .91448 -0.29713 Mabsadd L
.82 .87792 2.37764 .77254 Mabsadd L
.82 .87792 .56518 .7779 Mabsadd L
.82 .87792 0 2.5 Mabsadd L closepath F
.82095 .87793 0 2.5 Mabsadd m
.82095 .87793 -0.56518 .7779 Mabsadd L
.82095 .87793 -2.37764 .77254 Mabsadd L
.82095 .87793 -0.91448 -0.29713 Mabsadd L
.82095 .87793 -1.46946 -2.02254 Mabsadd L
.82095 .87793 0 -0.96154 Mabsadd L
.82095 .87793 1.46946 -2.02254 Mabsadd L
.82095 .87793 .91448 -0.29713 Mabsadd L
.82095 .87793 2.37764 .77254 Mabsadd L
.82095 .87793 .56518 .7779 Mabsadd L
.82095 .87793 0 2.5 Mabsadd L closepath F
.8219 .87907 0 2.5 Mabsadd m
.8219 .87907 -0.56518 .7779 Mabsadd L
.8219 .87907 -2.37764 .77254 Mabsadd L
.8219 .87907 -0.91448 -0.29713 Mabsadd L
.8219 .87907 -1.46946 -2.02254 Mabsadd L
.8219 .87907 0 -0.96154 Mabsadd L
.8219 .87907 1.46946 -2.02254 Mabsadd L
.8219 .87907 .91448 -0.29713 Mabsadd L
.8219 .87907 2.37764 .77254 Mabsadd L
.8219 .87907 .56518 .7779 Mabsadd L
.8219 .87907 0 2.5 Mabsadd L closepath F
.82286 .90142 0 2.5 Mabsadd m
.82286 .90142 -0.56518 .7779 Mabsadd L
.82286 .90142 -2.37764 .77254 Mabsadd L
.82286 .90142 -0.91448 -0.29713 Mabsadd L
.82286 .90142 -1.46946 -2.02254 Mabsadd L
.82286 .90142 0 -0.96154 Mabsadd L
.82286 .90142 1.46946 -2.02254 Mabsadd L
.82286 .90142 .91448 -0.29713 Mabsadd L
.82286 .90142 2.37764 .77254 Mabsadd L
.82286 .90142 .56518 .7779 Mabsadd L
.82286 .90142 0 2.5 Mabsadd L closepath F
.82381 .92881 0 2.5 Mabsadd m
.82381 .92881 -0.56518 .7779 Mabsadd L
.82381 .92881 -2.37764 .77254 Mabsadd L
.82381 .92881 -0.91448 -0.29713 Mabsadd L
.82381 .92881 -1.46946 -2.02254 Mabsadd L
.82381 .92881 0 -0.96154 Mabsadd L
.82381 .92881 1.46946 -2.02254 Mabsadd L
.82381 .92881 .91448 -0.29713 Mabsadd L
.82381 .92881 2.37764 .77254 Mabsadd L
.82381 .92881 .56518 .7779 Mabsadd L
.82381 .92881 0 2.5 Mabsadd L closepath F
.82476 .929 0 2.5 Mabsadd m
.82476 .929 -0.56518 .7779 Mabsadd L
.82476 .929 -2.37764 .77254 Mabsadd L
.82476 .929 -0.91448 -0.29713 Mabsadd L
.82476 .929 -1.46946 -2.02254 Mabsadd L
.82476 .929 0 -0.96154 Mabsadd L
.82476 .929 1.46946 -2.02254 Mabsadd L
.82476 .929 .91448 -0.29713 Mabsadd L
.82476 .929 2.37764 .77254 Mabsadd L
.82476 .929 .56518 .7779 Mabsadd L
.82476 .929 0 2.5 Mabsadd L closepath F
.82571 .9311 0 2.5 Mabsadd m
.82571 .9311 -0.56518 .7779 Mabsadd L
.82571 .9311 -2.37764 .77254 Mabsadd L
.82571 .9311 -0.91448 -0.29713 Mabsadd L
.82571 .9311 -1.46946 -2.02254 Mabsadd L
.82571 .9311 0 -0.96154 Mabsadd L
.82571 .9311 1.46946 -2.02254 Mabsadd L
.82571 .9311 .91448 -0.29713 Mabsadd L
.82571 .9311 2.37764 .77254 Mabsadd L
.82571 .9311 .56518 .7779 Mabsadd L
.82571 .9311 0 2.5 Mabsadd L closepath F
.82667 .93488 0 2.5 Mabsadd m
.82667 .93488 -0.56518 .7779 Mabsadd L
.82667 .93488 -2.37764 .77254 Mabsadd L
.82667 .93488 -0.91448 -0.29713 Mabsadd L
.82667 .93488 -1.46946 -2.02254 Mabsadd L
.82667 .93488 0 -0.96154 Mabsadd L
.82667 .93488 1.46946 -2.02254 Mabsadd L
.82667 .93488 .91448 -0.29713 Mabsadd L
.82667 .93488 2.37764 .77254 Mabsadd L
.82667 .93488 .56518 .7779 Mabsadd L
.82667 .93488 0 2.5 Mabsadd L closepath F
.82762 .93493 0 2.5 Mabsadd m
.82762 .93493 -0.56518 .7779 Mabsadd L
.82762 .93493 -2.37764 .77254 Mabsadd L
.82762 .93493 -0.91448 -0.29713 Mabsadd L
.82762 .93493 -1.46946 -2.02254 Mabsadd L
.82762 .93493 0 -0.96154 Mabsadd L
.82762 .93493 1.46946 -2.02254 Mabsadd L
.82762 .93493 .91448 -0.29713 Mabsadd L
.82762 .93493 2.37764 .77254 Mabsadd L
.82762 .93493 .56518 .7779 Mabsadd L
.82762 .93493 0 2.5 Mabsadd L closepath F
.82857 .93481 0 2.5 Mabsadd m
.82857 .93481 -0.56518 .7779 Mabsadd L
.82857 .93481 -2.37764 .77254 Mabsadd L
.82857 .93481 -0.91448 -0.29713 Mabsadd L
.82857 .93481 -1.46946 -2.02254 Mabsadd L
.82857 .93481 0 -0.96154 Mabsadd L
.82857 .93481 1.46946 -2.02254 Mabsadd L
.82857 .93481 .91448 -0.29713 Mabsadd L
.82857 .93481 2.37764 .77254 Mabsadd L
.82857 .93481 .56518 .7779 Mabsadd L
.82857 .93481 0 2.5 Mabsadd L closepath F
.82952 .9335 0 2.5 Mabsadd m
.82952 .9335 -0.56518 .7779 Mabsadd L
.82952 .9335 -2.37764 .77254 Mabsadd L
.82952 .9335 -0.91448 -0.29713 Mabsadd L
.82952 .9335 -1.46946 -2.02254 Mabsadd L
.82952 .9335 0 -0.96154 Mabsadd L
.82952 .9335 1.46946 -2.02254 Mabsadd L
.82952 .9335 .91448 -0.29713 Mabsadd L
.82952 .9335 2.37764 .77254 Mabsadd L
.82952 .9335 .56518 .7779 Mabsadd L
.82952 .9335 0 2.5 Mabsadd L closepath F
.83048 .9351 0 2.5 Mabsadd m
.83048 .9351 -0.56518 .7779 Mabsadd L
.83048 .9351 -2.37764 .77254 Mabsadd L
.83048 .9351 -0.91448 -0.29713 Mabsadd L
.83048 .9351 -1.46946 -2.02254 Mabsadd L
.83048 .9351 0 -0.96154 Mabsadd L
.83048 .9351 1.46946 -2.02254 Mabsadd L
.83048 .9351 .91448 -0.29713 Mabsadd L
.83048 .9351 2.37764 .77254 Mabsadd L
.83048 .9351 .56518 .7779 Mabsadd L
.83048 .9351 0 2.5 Mabsadd L closepath F
.83143 .9398 0 2.5 Mabsadd m
.83143 .9398 -0.56518 .7779 Mabsadd L
.83143 .9398 -2.37764 .77254 Mabsadd L
.83143 .9398 -0.91448 -0.29713 Mabsadd L
.83143 .9398 -1.46946 -2.02254 Mabsadd L
.83143 .9398 0 -0.96154 Mabsadd L
.83143 .9398 1.46946 -2.02254 Mabsadd L
.83143 .9398 .91448 -0.29713 Mabsadd L
.83143 .9398 2.37764 .77254 Mabsadd L
.83143 .9398 .56518 .7779 Mabsadd L
.83143 .9398 0 2.5 Mabsadd L closepath F
.83238 .94217 0 2.5 Mabsadd m
.83238 .94217 -0.56518 .7779 Mabsadd L
.83238 .94217 -2.37764 .77254 Mabsadd L
.83238 .94217 -0.91448 -0.29713 Mabsadd L
.83238 .94217 -1.46946 -2.02254 Mabsadd L
.83238 .94217 0 -0.96154 Mabsadd L
.83238 .94217 1.46946 -2.02254 Mabsadd L
.83238 .94217 .91448 -0.29713 Mabsadd L
.83238 .94217 2.37764 .77254 Mabsadd L
.83238 .94217 .56518 .7779 Mabsadd L
.83238 .94217 0 2.5 Mabsadd L closepath F
.83333 .96258 0 2.5 Mabsadd m
.83333 .96258 -0.56518 .7779 Mabsadd L
.83333 .96258 -2.37764 .77254 Mabsadd L
.83333 .96258 -0.91448 -0.29713 Mabsadd L
.83333 .96258 -1.46946 -2.02254 Mabsadd L
.83333 .96258 0 -0.96154 Mabsadd L
.83333 .96258 1.46946 -2.02254 Mabsadd L
.83333 .96258 .91448 -0.29713 Mabsadd L
.83333 .96258 2.37764 .77254 Mabsadd L
.83333 .96258 .56518 .7779 Mabsadd L
.83333 .96258 0 2.5 Mabsadd L closepath F
.83429 .96957 0 2.5 Mabsadd m
.83429 .96957 -0.56518 .7779 Mabsadd L
.83429 .96957 -2.37764 .77254 Mabsadd L
.83429 .96957 -0.91448 -0.29713 Mabsadd L
.83429 .96957 -1.46946 -2.02254 Mabsadd L
.83429 .96957 0 -0.96154 Mabsadd L
.83429 .96957 1.46946 -2.02254 Mabsadd L
.83429 .96957 .91448 -0.29713 Mabsadd L
.83429 .96957 2.37764 .77254 Mabsadd L
.83429 .96957 .56518 .7779 Mabsadd L
.83429 .96957 0 2.5 Mabsadd L closepath F
.83524 .97191 0 2.5 Mabsadd m
.83524 .97191 -0.56518 .7779 Mabsadd L
.83524 .97191 -2.37764 .77254 Mabsadd L
.83524 .97191 -0.91448 -0.29713 Mabsadd L
.83524 .97191 -1.46946 -2.02254 Mabsadd L
.83524 .97191 0 -0.96154 Mabsadd L
.83524 .97191 1.46946 -2.02254 Mabsadd L
.83524 .97191 .91448 -0.29713 Mabsadd L
.83524 .97191 2.37764 .77254 Mabsadd L
.83524 .97191 .56518 .7779 Mabsadd L
.83524 .97191 0 2.5 Mabsadd L closepath F
.83619 .97276 0 2.5 Mabsadd m
.83619 .97276 -0.56518 .7779 Mabsadd L
.83619 .97276 -2.37764 .77254 Mabsadd L
.83619 .97276 -0.91448 -0.29713 Mabsadd L
.83619 .97276 -1.46946 -2.02254 Mabsadd L
.83619 .97276 0 -0.96154 Mabsadd L
.83619 .97276 1.46946 -2.02254 Mabsadd L
.83619 .97276 .91448 -0.29713 Mabsadd L
.83619 .97276 2.37764 .77254 Mabsadd L
.83619 .97276 .56518 .7779 Mabsadd L
.83619 .97276 0 2.5 Mabsadd L closepath F
.83714 .97533 0 2.5 Mabsadd m
.83714 .97533 -0.56518 .7779 Mabsadd L
.83714 .97533 -2.37764 .77254 Mabsadd L
.83714 .97533 -0.91448 -0.29713 Mabsadd L
.83714 .97533 -1.46946 -2.02254 Mabsadd L
.83714 .97533 0 -0.96154 Mabsadd L
.83714 .97533 1.46946 -2.02254 Mabsadd L
.83714 .97533 .91448 -0.29713 Mabsadd L
.83714 .97533 2.37764 .77254 Mabsadd L
.83714 .97533 .56518 .7779 Mabsadd L
.83714 .97533 0 2.5 Mabsadd L closepath F
.8381 .97547 0 2.5 Mabsadd m
.8381 .97547 -0.56518 .7779 Mabsadd L
.8381 .97547 -2.37764 .77254 Mabsadd L
.8381 .97547 -0.91448 -0.29713 Mabsadd L
.8381 .97547 -1.46946 -2.02254 Mabsadd L
.8381 .97547 0 -0.96154 Mabsadd L
.8381 .97547 1.46946 -2.02254 Mabsadd L
.8381 .97547 .91448 -0.29713 Mabsadd L
.8381 .97547 2.37764 .77254 Mabsadd L
.8381 .97547 .56518 .7779 Mabsadd L
.8381 .97547 0 2.5 Mabsadd L closepath F
.83905 .97593 0 2.5 Mabsadd m
.83905 .97593 -0.56518 .7779 Mabsadd L
.83905 .97593 -2.37764 .77254 Mabsadd L
.83905 .97593 -0.91448 -0.29713 Mabsadd L
.83905 .97593 -1.46946 -2.02254 Mabsadd L
.83905 .97593 0 -0.96154 Mabsadd L
.83905 .97593 1.46946 -2.02254 Mabsadd L
.83905 .97593 .91448 -0.29713 Mabsadd L
.83905 .97593 2.37764 .77254 Mabsadd L
.83905 .97593 .56518 .7779 Mabsadd L
.83905 .97593 0 2.5 Mabsadd L closepath F
.84 .97596 0 2.5 Mabsadd m
.84 .97596 -0.56518 .7779 Mabsadd L
.84 .97596 -2.37764 .77254 Mabsadd L
.84 .97596 -0.91448 -0.29713 Mabsadd L
.84 .97596 -1.46946 -2.02254 Mabsadd L
.84 .97596 0 -0.96154 Mabsadd L
.84 .97596 1.46946 -2.02254 Mabsadd L
.84 .97596 .91448 -0.29713 Mabsadd L
.84 .97596 2.37764 .77254 Mabsadd L
.84 .97596 .56518 .7779 Mabsadd L
.84 .97596 0 2.5 Mabsadd L closepath F
.84095 .97607 0 2.5 Mabsadd m
.84095 .97607 -0.56518 .7779 Mabsadd L
.84095 .97607 -2.37764 .77254 Mabsadd L
.84095 .97607 -0.91448 -0.29713 Mabsadd L
.84095 .97607 -1.46946 -2.02254 Mabsadd L
.84095 .97607 0 -0.96154 Mabsadd L
.84095 .97607 1.46946 -2.02254 Mabsadd L
.84095 .97607 .91448 -0.29713 Mabsadd L
.84095 .97607 2.37764 .77254 Mabsadd L
.84095 .97607 .56518 .7779 Mabsadd L
.84095 .97607 0 2.5 Mabsadd L closepath F
.8419 .97608 0 2.5 Mabsadd m
.8419 .97608 -0.56518 .7779 Mabsadd L
.8419 .97608 -2.37764 .77254 Mabsadd L
.8419 .97608 -0.91448 -0.29713 Mabsadd L
.8419 .97608 -1.46946 -2.02254 Mabsadd L
.8419 .97608 0 -0.96154 Mabsadd L
.8419 .97608 1.46946 -2.02254 Mabsadd L
.8419 .97608 .91448 -0.29713 Mabsadd L
.8419 .97608 2.37764 .77254 Mabsadd L
.8419 .97608 .56518 .7779 Mabsadd L
.8419 .97608 0 2.5 Mabsadd L closepath F
.84286 .97611 0 2.5 Mabsadd m
.84286 .97611 -0.56518 .7779 Mabsadd L
.84286 .97611 -2.37764 .77254 Mabsadd L
.84286 .97611 -0.91448 -0.29713 Mabsadd L
.84286 .97611 -1.46946 -2.02254 Mabsadd L
.84286 .97611 0 -0.96154 Mabsadd L
.84286 .97611 1.46946 -2.02254 Mabsadd L
.84286 .97611 .91448 -0.29713 Mabsadd L
.84286 .97611 2.37764 .77254 Mabsadd L
.84286 .97611 .56518 .7779 Mabsadd L
.84286 .97611 0 2.5 Mabsadd L closepath F
.84381 .97616 0 2.5 Mabsadd m
.84381 .97616 -0.56518 .7779 Mabsadd L
.84381 .97616 -2.37764 .77254 Mabsadd L
.84381 .97616 -0.91448 -0.29713 Mabsadd L
.84381 .97616 -1.46946 -2.02254 Mabsadd L
.84381 .97616 0 -0.96154 Mabsadd L
.84381 .97616 1.46946 -2.02254 Mabsadd L
.84381 .97616 .91448 -0.29713 Mabsadd L
.84381 .97616 2.37764 .77254 Mabsadd L
.84381 .97616 .56518 .7779 Mabsadd L
.84381 .97616 0 2.5 Mabsadd L closepath F
.84476 .97616 0 2.5 Mabsadd m
.84476 .97616 -0.56518 .7779 Mabsadd L
.84476 .97616 -2.37764 .77254 Mabsadd L
.84476 .97616 -0.91448 -0.29713 Mabsadd L
.84476 .97616 -1.46946 -2.02254 Mabsadd L
.84476 .97616 0 -0.96154 Mabsadd L
.84476 .97616 1.46946 -2.02254 Mabsadd L
.84476 .97616 .91448 -0.29713 Mabsadd L
.84476 .97616 2.37764 .77254 Mabsadd L
.84476 .97616 .56518 .7779 Mabsadd L
.84476 .97616 0 2.5 Mabsadd L closepath F
.84571 .97616 0 2.5 Mabsadd m
.84571 .97616 -0.56518 .7779 Mabsadd L
.84571 .97616 -2.37764 .77254 Mabsadd L
.84571 .97616 -0.91448 -0.29713 Mabsadd L
.84571 .97616 -1.46946 -2.02254 Mabsadd L
.84571 .97616 0 -0.96154 Mabsadd L
.84571 .97616 1.46946 -2.02254 Mabsadd L
.84571 .97616 .91448 -0.29713 Mabsadd L
.84571 .97616 2.37764 .77254 Mabsadd L
.84571 .97616 .56518 .7779 Mabsadd L
.84571 .97616 0 2.5 Mabsadd L closepath F
.84667 .97617 0 2.5 Mabsadd m
.84667 .97617 -0.56518 .7779 Mabsadd L
.84667 .97617 -2.37764 .77254 Mabsadd L
.84667 .97617 -0.91448 -0.29713 Mabsadd L
.84667 .97617 -1.46946 -2.02254 Mabsadd L
.84667 .97617 0 -0.96154 Mabsadd L
.84667 .97617 1.46946 -2.02254 Mabsadd L
.84667 .97617 .91448 -0.29713 Mabsadd L
.84667 .97617 2.37764 .77254 Mabsadd L
.84667 .97617 .56518 .7779 Mabsadd L
.84667 .97617 0 2.5 Mabsadd L closepath F
.84762 .97617 0 2.5 Mabsadd m
.84762 .97617 -0.56518 .7779 Mabsadd L
.84762 .97617 -2.37764 .77254 Mabsadd L
.84762 .97617 -0.91448 -0.29713 Mabsadd L
.84762 .97617 -1.46946 -2.02254 Mabsadd L
.84762 .97617 0 -0.96154 Mabsadd L
.84762 .97617 1.46946 -2.02254 Mabsadd L
.84762 .97617 .91448 -0.29713 Mabsadd L
.84762 .97617 2.37764 .77254 Mabsadd L
.84762 .97617 .56518 .7779 Mabsadd L
.84762 .97617 0 2.5 Mabsadd L closepath F
.84857 .97618 0 2.5 Mabsadd m
.84857 .97618 -0.56518 .7779 Mabsadd L
.84857 .97618 -2.37764 .77254 Mabsadd L
.84857 .97618 -0.91448 -0.29713 Mabsadd L
.84857 .97618 -1.46946 -2.02254 Mabsadd L
.84857 .97618 0 -0.96154 Mabsadd L
.84857 .97618 1.46946 -2.02254 Mabsadd L
.84857 .97618 .91448 -0.29713 Mabsadd L
.84857 .97618 2.37764 .77254 Mabsadd L
.84857 .97618 .56518 .7779 Mabsadd L
.84857 .97618 0 2.5 Mabsadd L closepath F
.84952 .97618 0 2.5 Mabsadd m
.84952 .97618 -0.56518 .7779 Mabsadd L
.84952 .97618 -2.37764 .77254 Mabsadd L
.84952 .97618 -0.91448 -0.29713 Mabsadd L
.84952 .97618 -1.46946 -2.02254 Mabsadd L
.84952 .97618 0 -0.96154 Mabsadd L
.84952 .97618 1.46946 -2.02254 Mabsadd L
.84952 .97618 .91448 -0.29713 Mabsadd L
.84952 .97618 2.37764 .77254 Mabsadd L
.84952 .97618 .56518 .7779 Mabsadd L
.84952 .97618 0 2.5 Mabsadd L closepath F
.85048 .97618 0 2.5 Mabsadd m
.85048 .97618 -0.56518 .7779 Mabsadd L
.85048 .97618 -2.37764 .77254 Mabsadd L
.85048 .97618 -0.91448 -0.29713 Mabsadd L
.85048 .97618 -1.46946 -2.02254 Mabsadd L
.85048 .97618 0 -0.96154 Mabsadd L
.85048 .97618 1.46946 -2.02254 Mabsadd L
.85048 .97618 .91448 -0.29713 Mabsadd L
.85048 .97618 2.37764 .77254 Mabsadd L
.85048 .97618 .56518 .7779 Mabsadd L
.85048 .97618 0 2.5 Mabsadd L closepath F
.85143 .97618 0 2.5 Mabsadd m
.85143 .97618 -0.56518 .7779 Mabsadd L
.85143 .97618 -2.37764 .77254 Mabsadd L
.85143 .97618 -0.91448 -0.29713 Mabsadd L
.85143 .97618 -1.46946 -2.02254 Mabsadd L
.85143 .97618 0 -0.96154 Mabsadd L
.85143 .97618 1.46946 -2.02254 Mabsadd L
.85143 .97618 .91448 -0.29713 Mabsadd L
.85143 .97618 2.37764 .77254 Mabsadd L
.85143 .97618 .56518 .7779 Mabsadd L
.85143 .97618 0 2.5 Mabsadd L closepath F
.85238 .97619 0 2.5 Mabsadd m
.85238 .97619 -0.56518 .7779 Mabsadd L
.85238 .97619 -2.37764 .77254 Mabsadd L
.85238 .97619 -0.91448 -0.29713 Mabsadd L
.85238 .97619 -1.46946 -2.02254 Mabsadd L
.85238 .97619 0 -0.96154 Mabsadd L
.85238 .97619 1.46946 -2.02254 Mabsadd L
.85238 .97619 .91448 -0.29713 Mabsadd L
.85238 .97619 2.37764 .77254 Mabsadd L
.85238 .97619 .56518 .7779 Mabsadd L
.85238 .97619 0 2.5 Mabsadd L closepath F
.85333 .97619 0 2.5 Mabsadd m
.85333 .97619 -0.56518 .7779 Mabsadd L
.85333 .97619 -2.37764 .77254 Mabsadd L
.85333 .97619 -0.91448 -0.29713 Mabsadd L
.85333 .97619 -1.46946 -2.02254 Mabsadd L
.85333 .97619 0 -0.96154 Mabsadd L
.85333 .97619 1.46946 -2.02254 Mabsadd L
.85333 .97619 .91448 -0.29713 Mabsadd L
.85333 .97619 2.37764 .77254 Mabsadd L
.85333 .97619 .56518 .7779 Mabsadd L
.85333 .97619 0 2.5 Mabsadd L closepath F
.85429 .97619 0 2.5 Mabsadd m
.85429 .97619 -0.56518 .7779 Mabsadd L
.85429 .97619 -2.37764 .77254 Mabsadd L
.85429 .97619 -0.91448 -0.29713 Mabsadd L
.85429 .97619 -1.46946 -2.02254 Mabsadd L
.85429 .97619 0 -0.96154 Mabsadd L
.85429 .97619 1.46946 -2.02254 Mabsadd L
.85429 .97619 .91448 -0.29713 Mabsadd L
.85429 .97619 2.37764 .77254 Mabsadd L
.85429 .97619 .56518 .7779 Mabsadd L
.85429 .97619 0 2.5 Mabsadd L closepath F
.85524 .97618 0 2.5 Mabsadd m
.85524 .97618 -0.56518 .7779 Mabsadd L
.85524 .97618 -2.37764 .77254 Mabsadd L
.85524 .97618 -0.91448 -0.29713 Mabsadd L
.85524 .97618 -1.46946 -2.02254 Mabsadd L
.85524 .97618 0 -0.96154 Mabsadd L
.85524 .97618 1.46946 -2.02254 Mabsadd L
.85524 .97618 .91448 -0.29713 Mabsadd L
.85524 .97618 2.37764 .77254 Mabsadd L
.85524 .97618 .56518 .7779 Mabsadd L
.85524 .97618 0 2.5 Mabsadd L closepath F
.85619 .97617 0 2.5 Mabsadd m
.85619 .97617 -0.56518 .7779 Mabsadd L
.85619 .97617 -2.37764 .77254 Mabsadd L
.85619 .97617 -0.91448 -0.29713 Mabsadd L
.85619 .97617 -1.46946 -2.02254 Mabsadd L
.85619 .97617 0 -0.96154 Mabsadd L
.85619 .97617 1.46946 -2.02254 Mabsadd L
.85619 .97617 .91448 -0.29713 Mabsadd L
.85619 .97617 2.37764 .77254 Mabsadd L
.85619 .97617 .56518 .7779 Mabsadd L
.85619 .97617 0 2.5 Mabsadd L closepath F
.85714 .97617 0 2.5 Mabsadd m
.85714 .97617 -0.56518 .7779 Mabsadd L
.85714 .97617 -2.37764 .77254 Mabsadd L
.85714 .97617 -0.91448 -0.29713 Mabsadd L
.85714 .97617 -1.46946 -2.02254 Mabsadd L
.85714 .97617 0 -0.96154 Mabsadd L
.85714 .97617 1.46946 -2.02254 Mabsadd L
.85714 .97617 .91448 -0.29713 Mabsadd L
.85714 .97617 2.37764 .77254 Mabsadd L
.85714 .97617 .56518 .7779 Mabsadd L
.85714 .97617 0 2.5 Mabsadd L closepath F
.8581 .97617 0 2.5 Mabsadd m
.8581 .97617 -0.56518 .7779 Mabsadd L
.8581 .97617 -2.37764 .77254 Mabsadd L
.8581 .97617 -0.91448 -0.29713 Mabsadd L
.8581 .97617 -1.46946 -2.02254 Mabsadd L
.8581 .97617 0 -0.96154 Mabsadd L
.8581 .97617 1.46946 -2.02254 Mabsadd L
.8581 .97617 .91448 -0.29713 Mabsadd L
.8581 .97617 2.37764 .77254 Mabsadd L
.8581 .97617 .56518 .7779 Mabsadd L
.8581 .97617 0 2.5 Mabsadd L closepath F
.85905 .97616 0 2.5 Mabsadd m
.85905 .97616 -0.56518 .7779 Mabsadd L
.85905 .97616 -2.37764 .77254 Mabsadd L
.85905 .97616 -0.91448 -0.29713 Mabsadd L
.85905 .97616 -1.46946 -2.02254 Mabsadd L
.85905 .97616 0 -0.96154 Mabsadd L
.85905 .97616 1.46946 -2.02254 Mabsadd L
.85905 .97616 .91448 -0.29713 Mabsadd L
.85905 .97616 2.37764 .77254 Mabsadd L
.85905 .97616 .56518 .7779 Mabsadd L
.85905 .97616 0 2.5 Mabsadd L closepath F
.86 .97615 0 2.5 Mabsadd m
.86 .97615 -0.56518 .7779 Mabsadd L
.86 .97615 -2.37764 .77254 Mabsadd L
.86 .97615 -0.91448 -0.29713 Mabsadd L
.86 .97615 -1.46946 -2.02254 Mabsadd L
.86 .97615 0 -0.96154 Mabsadd L
.86 .97615 1.46946 -2.02254 Mabsadd L
.86 .97615 .91448 -0.29713 Mabsadd L
.86 .97615 2.37764 .77254 Mabsadd L
.86 .97615 .56518 .7779 Mabsadd L
.86 .97615 0 2.5 Mabsadd L closepath F
.86095 .97615 0 2.5 Mabsadd m
.86095 .97615 -0.56518 .7779 Mabsadd L
.86095 .97615 -2.37764 .77254 Mabsadd L
.86095 .97615 -0.91448 -0.29713 Mabsadd L
.86095 .97615 -1.46946 -2.02254 Mabsadd L
.86095 .97615 0 -0.96154 Mabsadd L
.86095 .97615 1.46946 -2.02254 Mabsadd L
.86095 .97615 .91448 -0.29713 Mabsadd L
.86095 .97615 2.37764 .77254 Mabsadd L
.86095 .97615 .56518 .7779 Mabsadd L
.86095 .97615 0 2.5 Mabsadd L closepath F
.8619 .97614 0 2.5 Mabsadd m
.8619 .97614 -0.56518 .7779 Mabsadd L
.8619 .97614 -2.37764 .77254 Mabsadd L
.8619 .97614 -0.91448 -0.29713 Mabsadd L
.8619 .97614 -1.46946 -2.02254 Mabsadd L
.8619 .97614 0 -0.96154 Mabsadd L
.8619 .97614 1.46946 -2.02254 Mabsadd L
.8619 .97614 .91448 -0.29713 Mabsadd L
.8619 .97614 2.37764 .77254 Mabsadd L
.8619 .97614 .56518 .7779 Mabsadd L
.8619 .97614 0 2.5 Mabsadd L closepath F
.86286 .97612 0 2.5 Mabsadd m
.86286 .97612 -0.56518 .7779 Mabsadd L
.86286 .97612 -2.37764 .77254 Mabsadd L
.86286 .97612 -0.91448 -0.29713 Mabsadd L
.86286 .97612 -1.46946 -2.02254 Mabsadd L
.86286 .97612 0 -0.96154 Mabsadd L
.86286 .97612 1.46946 -2.02254 Mabsadd L
.86286 .97612 .91448 -0.29713 Mabsadd L
.86286 .97612 2.37764 .77254 Mabsadd L
.86286 .97612 .56518 .7779 Mabsadd L
.86286 .97612 0 2.5 Mabsadd L closepath F
.86381 .97611 0 2.5 Mabsadd m
.86381 .97611 -0.56518 .7779 Mabsadd L
.86381 .97611 -2.37764 .77254 Mabsadd L
.86381 .97611 -0.91448 -0.29713 Mabsadd L
.86381 .97611 -1.46946 -2.02254 Mabsadd L
.86381 .97611 0 -0.96154 Mabsadd L
.86381 .97611 1.46946 -2.02254 Mabsadd L
.86381 .97611 .91448 -0.29713 Mabsadd L
.86381 .97611 2.37764 .77254 Mabsadd L
.86381 .97611 .56518 .7779 Mabsadd L
.86381 .97611 0 2.5 Mabsadd L closepath F
.86476 .97608 0 2.5 Mabsadd m
.86476 .97608 -0.56518 .7779 Mabsadd L
.86476 .97608 -2.37764 .77254 Mabsadd L
.86476 .97608 -0.91448 -0.29713 Mabsadd L
.86476 .97608 -1.46946 -2.02254 Mabsadd L
.86476 .97608 0 -0.96154 Mabsadd L
.86476 .97608 1.46946 -2.02254 Mabsadd L
.86476 .97608 .91448 -0.29713 Mabsadd L
.86476 .97608 2.37764 .77254 Mabsadd L
.86476 .97608 .56518 .7779 Mabsadd L
.86476 .97608 0 2.5 Mabsadd L closepath F
.86571 .97607 0 2.5 Mabsadd m
.86571 .97607 -0.56518 .7779 Mabsadd L
.86571 .97607 -2.37764 .77254 Mabsadd L
.86571 .97607 -0.91448 -0.29713 Mabsadd L
.86571 .97607 -1.46946 -2.02254 Mabsadd L
.86571 .97607 0 -0.96154 Mabsadd L
.86571 .97607 1.46946 -2.02254 Mabsadd L
.86571 .97607 .91448 -0.29713 Mabsadd L
.86571 .97607 2.37764 .77254 Mabsadd L
.86571 .97607 .56518 .7779 Mabsadd L
.86571 .97607 0 2.5 Mabsadd L closepath F
.86667 .97594 0 2.5 Mabsadd m
.86667 .97594 -0.56518 .7779 Mabsadd L
.86667 .97594 -2.37764 .77254 Mabsadd L
.86667 .97594 -0.91448 -0.29713 Mabsadd L
.86667 .97594 -1.46946 -2.02254 Mabsadd L
.86667 .97594 0 -0.96154 Mabsadd L
.86667 .97594 1.46946 -2.02254 Mabsadd L
.86667 .97594 .91448 -0.29713 Mabsadd L
.86667 .97594 2.37764 .77254 Mabsadd L
.86667 .97594 .56518 .7779 Mabsadd L
.86667 .97594 0 2.5 Mabsadd L closepath F
.86762 .97588 0 2.5 Mabsadd m
.86762 .97588 -0.56518 .7779 Mabsadd L
.86762 .97588 -2.37764 .77254 Mabsadd L
.86762 .97588 -0.91448 -0.29713 Mabsadd L
.86762 .97588 -1.46946 -2.02254 Mabsadd L
.86762 .97588 0 -0.96154 Mabsadd L
.86762 .97588 1.46946 -2.02254 Mabsadd L
.86762 .97588 .91448 -0.29713 Mabsadd L
.86762 .97588 2.37764 .77254 Mabsadd L
.86762 .97588 .56518 .7779 Mabsadd L
.86762 .97588 0 2.5 Mabsadd L closepath F
.86857 .97529 0 2.5 Mabsadd m
.86857 .97529 -0.56518 .7779 Mabsadd L
.86857 .97529 -2.37764 .77254 Mabsadd L
.86857 .97529 -0.91448 -0.29713 Mabsadd L
.86857 .97529 -1.46946 -2.02254 Mabsadd L
.86857 .97529 0 -0.96154 Mabsadd L
.86857 .97529 1.46946 -2.02254 Mabsadd L
.86857 .97529 .91448 -0.29713 Mabsadd L
.86857 .97529 2.37764 .77254 Mabsadd L
.86857 .97529 .56518 .7779 Mabsadd L
.86857 .97529 0 2.5 Mabsadd L closepath F
.86952 .97497 0 2.5 Mabsadd m
.86952 .97497 -0.56518 .7779 Mabsadd L
.86952 .97497 -2.37764 .77254 Mabsadd L
.86952 .97497 -0.91448 -0.29713 Mabsadd L
.86952 .97497 -1.46946 -2.02254 Mabsadd L
.86952 .97497 0 -0.96154 Mabsadd L
.86952 .97497 1.46946 -2.02254 Mabsadd L
.86952 .97497 .91448 -0.29713 Mabsadd L
.86952 .97497 2.37764 .77254 Mabsadd L
.86952 .97497 .56518 .7779 Mabsadd L
.86952 .97497 0 2.5 Mabsadd L closepath F
.87048 .97149 0 2.5 Mabsadd m
.87048 .97149 -0.56518 .7779 Mabsadd L
.87048 .97149 -2.37764 .77254 Mabsadd L
.87048 .97149 -0.91448 -0.29713 Mabsadd L
.87048 .97149 -1.46946 -2.02254 Mabsadd L
.87048 .97149 0 -0.96154 Mabsadd L
.87048 .97149 1.46946 -2.02254 Mabsadd L
.87048 .97149 .91448 -0.29713 Mabsadd L
.87048 .97149 2.37764 .77254 Mabsadd L
.87048 .97149 .56518 .7779 Mabsadd L
.87048 .97149 0 2.5 Mabsadd L closepath F
.87143 .96908 0 2.5 Mabsadd m
.87143 .96908 -0.56518 .7779 Mabsadd L
.87143 .96908 -2.37764 .77254 Mabsadd L
.87143 .96908 -0.91448 -0.29713 Mabsadd L
.87143 .96908 -1.46946 -2.02254 Mabsadd L
.87143 .96908 0 -0.96154 Mabsadd L
.87143 .96908 1.46946 -2.02254 Mabsadd L
.87143 .96908 .91448 -0.29713 Mabsadd L
.87143 .96908 2.37764 .77254 Mabsadd L
.87143 .96908 .56518 .7779 Mabsadd L
.87143 .96908 0 2.5 Mabsadd L closepath F
.87238 .94013 0 2.5 Mabsadd m
.87238 .94013 -0.56518 .7779 Mabsadd L
.87238 .94013 -2.37764 .77254 Mabsadd L
.87238 .94013 -0.91448 -0.29713 Mabsadd L
.87238 .94013 -1.46946 -2.02254 Mabsadd L
.87238 .94013 0 -0.96154 Mabsadd L
.87238 .94013 1.46946 -2.02254 Mabsadd L
.87238 .94013 .91448 -0.29713 Mabsadd L
.87238 .94013 2.37764 .77254 Mabsadd L
.87238 .94013 .56518 .7779 Mabsadd L
.87238 .94013 0 2.5 Mabsadd L closepath F
.87333 .91388 0 2.5 Mabsadd m
.87333 .91388 -0.56518 .7779 Mabsadd L
.87333 .91388 -2.37764 .77254 Mabsadd L
.87333 .91388 -0.91448 -0.29713 Mabsadd L
.87333 .91388 -1.46946 -2.02254 Mabsadd L
.87333 .91388 0 -0.96154 Mabsadd L
.87333 .91388 1.46946 -2.02254 Mabsadd L
.87333 .91388 .91448 -0.29713 Mabsadd L
.87333 .91388 2.37764 .77254 Mabsadd L
.87333 .91388 .56518 .7779 Mabsadd L
.87333 .91388 0 2.5 Mabsadd L closepath F
.87429 .91389 0 2.5 Mabsadd m
.87429 .91389 -0.56518 .7779 Mabsadd L
.87429 .91389 -2.37764 .77254 Mabsadd L
.87429 .91389 -0.91448 -0.29713 Mabsadd L
.87429 .91389 -1.46946 -2.02254 Mabsadd L
.87429 .91389 0 -0.96154 Mabsadd L
.87429 .91389 1.46946 -2.02254 Mabsadd L
.87429 .91389 .91448 -0.29713 Mabsadd L
.87429 .91389 2.37764 .77254 Mabsadd L
.87429 .91389 .56518 .7779 Mabsadd L
.87429 .91389 0 2.5 Mabsadd L closepath F
.87524 .91473 0 2.5 Mabsadd m
.87524 .91473 -0.56518 .7779 Mabsadd L
.87524 .91473 -2.37764 .77254 Mabsadd L
.87524 .91473 -0.91448 -0.29713 Mabsadd L
.87524 .91473 -1.46946 -2.02254 Mabsadd L
.87524 .91473 0 -0.96154 Mabsadd L
.87524 .91473 1.46946 -2.02254 Mabsadd L
.87524 .91473 .91448 -0.29713 Mabsadd L
.87524 .91473 2.37764 .77254 Mabsadd L
.87524 .91473 .56518 .7779 Mabsadd L
.87524 .91473 0 2.5 Mabsadd L closepath F
.87619 .93144 0 2.5 Mabsadd m
.87619 .93144 -0.56518 .7779 Mabsadd L
.87619 .93144 -2.37764 .77254 Mabsadd L
.87619 .93144 -0.91448 -0.29713 Mabsadd L
.87619 .93144 -1.46946 -2.02254 Mabsadd L
.87619 .93144 0 -0.96154 Mabsadd L
.87619 .93144 1.46946 -2.02254 Mabsadd L
.87619 .93144 .91448 -0.29713 Mabsadd L
.87619 .93144 2.37764 .77254 Mabsadd L
.87619 .93144 .56518 .7779 Mabsadd L
.87619 .93144 0 2.5 Mabsadd L closepath F
.87714 .95113 0 2.5 Mabsadd m
.87714 .95113 -0.56518 .7779 Mabsadd L
.87714 .95113 -2.37764 .77254 Mabsadd L
.87714 .95113 -0.91448 -0.29713 Mabsadd L
.87714 .95113 -1.46946 -2.02254 Mabsadd L
.87714 .95113 0 -0.96154 Mabsadd L
.87714 .95113 1.46946 -2.02254 Mabsadd L
.87714 .95113 .91448 -0.29713 Mabsadd L
.87714 .95113 2.37764 .77254 Mabsadd L
.87714 .95113 .56518 .7779 Mabsadd L
.87714 .95113 0 2.5 Mabsadd L closepath F
.8781 .95126 0 2.5 Mabsadd m
.8781 .95126 -0.56518 .7779 Mabsadd L
.8781 .95126 -2.37764 .77254 Mabsadd L
.8781 .95126 -0.91448 -0.29713 Mabsadd L
.8781 .95126 -1.46946 -2.02254 Mabsadd L
.8781 .95126 0 -0.96154 Mabsadd L
.8781 .95126 1.46946 -2.02254 Mabsadd L
.8781 .95126 .91448 -0.29713 Mabsadd L
.8781 .95126 2.37764 .77254 Mabsadd L
.8781 .95126 .56518 .7779 Mabsadd L
.8781 .95126 0 2.5 Mabsadd L closepath F
.87905 .95293 0 2.5 Mabsadd m
.87905 .95293 -0.56518 .7779 Mabsadd L
.87905 .95293 -2.37764 .77254 Mabsadd L
.87905 .95293 -0.91448 -0.29713 Mabsadd L
.87905 .95293 -1.46946 -2.02254 Mabsadd L
.87905 .95293 0 -0.96154 Mabsadd L
.87905 .95293 1.46946 -2.02254 Mabsadd L
.87905 .95293 .91448 -0.29713 Mabsadd L
.87905 .95293 2.37764 .77254 Mabsadd L
.87905 .95293 .56518 .7779 Mabsadd L
.87905 .95293 0 2.5 Mabsadd L closepath F
.88 .95695 0 2.5 Mabsadd m
.88 .95695 -0.56518 .7779 Mabsadd L
.88 .95695 -2.37764 .77254 Mabsadd L
.88 .95695 -0.91448 -0.29713 Mabsadd L
.88 .95695 -1.46946 -2.02254 Mabsadd L
.88 .95695 0 -0.96154 Mabsadd L
.88 .95695 1.46946 -2.02254 Mabsadd L
.88 .95695 .91448 -0.29713 Mabsadd L
.88 .95695 2.37764 .77254 Mabsadd L
.88 .95695 .56518 .7779 Mabsadd L
.88 .95695 0 2.5 Mabsadd L closepath F
.88095 .95729 0 2.5 Mabsadd m
.88095 .95729 -0.56518 .7779 Mabsadd L
.88095 .95729 -2.37764 .77254 Mabsadd L
.88095 .95729 -0.91448 -0.29713 Mabsadd L
.88095 .95729 -1.46946 -2.02254 Mabsadd L
.88095 .95729 0 -0.96154 Mabsadd L
.88095 .95729 1.46946 -2.02254 Mabsadd L
.88095 .95729 .91448 -0.29713 Mabsadd L
.88095 .95729 2.37764 .77254 Mabsadd L
.88095 .95729 .56518 .7779 Mabsadd L
.88095 .95729 0 2.5 Mabsadd L closepath F
.8819 .95713 0 2.5 Mabsadd m
.8819 .95713 -0.56518 .7779 Mabsadd L
.8819 .95713 -2.37764 .77254 Mabsadd L
.8819 .95713 -0.91448 -0.29713 Mabsadd L
.8819 .95713 -1.46946 -2.02254 Mabsadd L
.8819 .95713 0 -0.96154 Mabsadd L
.8819 .95713 1.46946 -2.02254 Mabsadd L
.8819 .95713 .91448 -0.29713 Mabsadd L
.8819 .95713 2.37764 .77254 Mabsadd L
.8819 .95713 .56518 .7779 Mabsadd L
.8819 .95713 0 2.5 Mabsadd L closepath F
.88286 .95517 0 2.5 Mabsadd m
.88286 .95517 -0.56518 .7779 Mabsadd L
.88286 .95517 -2.37764 .77254 Mabsadd L
.88286 .95517 -0.91448 -0.29713 Mabsadd L
.88286 .95517 -1.46946 -2.02254 Mabsadd L
.88286 .95517 0 -0.96154 Mabsadd L
.88286 .95517 1.46946 -2.02254 Mabsadd L
.88286 .95517 .91448 -0.29713 Mabsadd L
.88286 .95517 2.37764 .77254 Mabsadd L
.88286 .95517 .56518 .7779 Mabsadd L
.88286 .95517 0 2.5 Mabsadd L closepath F
.88381 .95737 0 2.5 Mabsadd m
.88381 .95737 -0.56518 .7779 Mabsadd L
.88381 .95737 -2.37764 .77254 Mabsadd L
.88381 .95737 -0.91448 -0.29713 Mabsadd L
.88381 .95737 -1.46946 -2.02254 Mabsadd L
.88381 .95737 0 -0.96154 Mabsadd L
.88381 .95737 1.46946 -2.02254 Mabsadd L
.88381 .95737 .91448 -0.29713 Mabsadd L
.88381 .95737 2.37764 .77254 Mabsadd L
.88381 .95737 .56518 .7779 Mabsadd L
.88381 .95737 0 2.5 Mabsadd L closepath F
.88476 .96398 0 2.5 Mabsadd m
.88476 .96398 -0.56518 .7779 Mabsadd L
.88476 .96398 -2.37764 .77254 Mabsadd L
.88476 .96398 -0.91448 -0.29713 Mabsadd L
.88476 .96398 -1.46946 -2.02254 Mabsadd L
.88476 .96398 0 -0.96154 Mabsadd L
.88476 .96398 1.46946 -2.02254 Mabsadd L
.88476 .96398 .91448 -0.29713 Mabsadd L
.88476 .96398 2.37764 .77254 Mabsadd L
.88476 .96398 .56518 .7779 Mabsadd L
.88476 .96398 0 2.5 Mabsadd L closepath F
.88571 .964 0 2.5 Mabsadd m
.88571 .964 -0.56518 .7779 Mabsadd L
.88571 .964 -2.37764 .77254 Mabsadd L
.88571 .964 -0.91448 -0.29713 Mabsadd L
.88571 .964 -1.46946 -2.02254 Mabsadd L
.88571 .964 0 -0.96154 Mabsadd L
.88571 .964 1.46946 -2.02254 Mabsadd L
.88571 .964 .91448 -0.29713 Mabsadd L
.88571 .964 2.37764 .77254 Mabsadd L
.88571 .964 .56518 .7779 Mabsadd L
.88571 .964 0 2.5 Mabsadd L closepath F
.88667 .96405 0 2.5 Mabsadd m
.88667 .96405 -0.56518 .7779 Mabsadd L
.88667 .96405 -2.37764 .77254 Mabsadd L
.88667 .96405 -0.91448 -0.29713 Mabsadd L
.88667 .96405 -1.46946 -2.02254 Mabsadd L
.88667 .96405 0 -0.96154 Mabsadd L
.88667 .96405 1.46946 -2.02254 Mabsadd L
.88667 .96405 .91448 -0.29713 Mabsadd L
.88667 .96405 2.37764 .77254 Mabsadd L
.88667 .96405 .56518 .7779 Mabsadd L
.88667 .96405 0 2.5 Mabsadd L closepath F
.88762 .96406 0 2.5 Mabsadd m
.88762 .96406 -0.56518 .7779 Mabsadd L
.88762 .96406 -2.37764 .77254 Mabsadd L
.88762 .96406 -0.91448 -0.29713 Mabsadd L
.88762 .96406 -1.46946 -2.02254 Mabsadd L
.88762 .96406 0 -0.96154 Mabsadd L
.88762 .96406 1.46946 -2.02254 Mabsadd L
.88762 .96406 .91448 -0.29713 Mabsadd L
.88762 .96406 2.37764 .77254 Mabsadd L
.88762 .96406 .56518 .7779 Mabsadd L
.88762 .96406 0 2.5 Mabsadd L closepath F
.88857 .96403 0 2.5 Mabsadd m
.88857 .96403 -0.56518 .7779 Mabsadd L
.88857 .96403 -2.37764 .77254 Mabsadd L
.88857 .96403 -0.91448 -0.29713 Mabsadd L
.88857 .96403 -1.46946 -2.02254 Mabsadd L
.88857 .96403 0 -0.96154 Mabsadd L
.88857 .96403 1.46946 -2.02254 Mabsadd L
.88857 .96403 .91448 -0.29713 Mabsadd L
.88857 .96403 2.37764 .77254 Mabsadd L
.88857 .96403 .56518 .7779 Mabsadd L
.88857 .96403 0 2.5 Mabsadd L closepath F
.88952 .96397 0 2.5 Mabsadd m
.88952 .96397 -0.56518 .7779 Mabsadd L
.88952 .96397 -2.37764 .77254 Mabsadd L
.88952 .96397 -0.91448 -0.29713 Mabsadd L
.88952 .96397 -1.46946 -2.02254 Mabsadd L
.88952 .96397 0 -0.96154 Mabsadd L
.88952 .96397 1.46946 -2.02254 Mabsadd L
.88952 .96397 .91448 -0.29713 Mabsadd L
.88952 .96397 2.37764 .77254 Mabsadd L
.88952 .96397 .56518 .7779 Mabsadd L
.88952 .96397 0 2.5 Mabsadd L closepath F
.89048 .9639 0 2.5 Mabsadd m
.89048 .9639 -0.56518 .7779 Mabsadd L
.89048 .9639 -2.37764 .77254 Mabsadd L
.89048 .9639 -0.91448 -0.29713 Mabsadd L
.89048 .9639 -1.46946 -2.02254 Mabsadd L
.89048 .9639 0 -0.96154 Mabsadd L
.89048 .9639 1.46946 -2.02254 Mabsadd L
.89048 .9639 .91448 -0.29713 Mabsadd L
.89048 .9639 2.37764 .77254 Mabsadd L
.89048 .9639 .56518 .7779 Mabsadd L
.89048 .9639 0 2.5 Mabsadd L closepath F
.89143 .96406 0 2.5 Mabsadd m
.89143 .96406 -0.56518 .7779 Mabsadd L
.89143 .96406 -2.37764 .77254 Mabsadd L
.89143 .96406 -0.91448 -0.29713 Mabsadd L
.89143 .96406 -1.46946 -2.02254 Mabsadd L
.89143 .96406 0 -0.96154 Mabsadd L
.89143 .96406 1.46946 -2.02254 Mabsadd L
.89143 .96406 .91448 -0.29713 Mabsadd L
.89143 .96406 2.37764 .77254 Mabsadd L
.89143 .96406 .56518 .7779 Mabsadd L
.89143 .96406 0 2.5 Mabsadd L closepath F
.89238 .96457 0 2.5 Mabsadd m
.89238 .96457 -0.56518 .7779 Mabsadd L
.89238 .96457 -2.37764 .77254 Mabsadd L
.89238 .96457 -0.91448 -0.29713 Mabsadd L
.89238 .96457 -1.46946 -2.02254 Mabsadd L
.89238 .96457 0 -0.96154 Mabsadd L
.89238 .96457 1.46946 -2.02254 Mabsadd L
.89238 .96457 .91448 -0.29713 Mabsadd L
.89238 .96457 2.37764 .77254 Mabsadd L
.89238 .96457 .56518 .7779 Mabsadd L
.89238 .96457 0 2.5 Mabsadd L closepath F
.89333 .9646 0 2.5 Mabsadd m
.89333 .9646 -0.56518 .7779 Mabsadd L
.89333 .9646 -2.37764 .77254 Mabsadd L
.89333 .9646 -0.91448 -0.29713 Mabsadd L
.89333 .9646 -1.46946 -2.02254 Mabsadd L
.89333 .9646 0 -0.96154 Mabsadd L
.89333 .9646 1.46946 -2.02254 Mabsadd L
.89333 .9646 .91448 -0.29713 Mabsadd L
.89333 .9646 2.37764 .77254 Mabsadd L
.89333 .9646 .56518 .7779 Mabsadd L
.89333 .9646 0 2.5 Mabsadd L closepath F
.89429 .96472 0 2.5 Mabsadd m
.89429 .96472 -0.56518 .7779 Mabsadd L
.89429 .96472 -2.37764 .77254 Mabsadd L
.89429 .96472 -0.91448 -0.29713 Mabsadd L
.89429 .96472 -1.46946 -2.02254 Mabsadd L
.89429 .96472 0 -0.96154 Mabsadd L
.89429 .96472 1.46946 -2.02254 Mabsadd L
.89429 .96472 .91448 -0.29713 Mabsadd L
.89429 .96472 2.37764 .77254 Mabsadd L
.89429 .96472 .56518 .7779 Mabsadd L
.89429 .96472 0 2.5 Mabsadd L closepath F
.89524 .96472 0 2.5 Mabsadd m
.89524 .96472 -0.56518 .7779 Mabsadd L
.89524 .96472 -2.37764 .77254 Mabsadd L
.89524 .96472 -0.91448 -0.29713 Mabsadd L
.89524 .96472 -1.46946 -2.02254 Mabsadd L
.89524 .96472 0 -0.96154 Mabsadd L
.89524 .96472 1.46946 -2.02254 Mabsadd L
.89524 .96472 .91448 -0.29713 Mabsadd L
.89524 .96472 2.37764 .77254 Mabsadd L
.89524 .96472 .56518 .7779 Mabsadd L
.89524 .96472 0 2.5 Mabsadd L closepath F
.89619 .96476 0 2.5 Mabsadd m
.89619 .96476 -0.56518 .7779 Mabsadd L
.89619 .96476 -2.37764 .77254 Mabsadd L
.89619 .96476 -0.91448 -0.29713 Mabsadd L
.89619 .96476 -1.46946 -2.02254 Mabsadd L
.89619 .96476 0 -0.96154 Mabsadd L
.89619 .96476 1.46946 -2.02254 Mabsadd L
.89619 .96476 .91448 -0.29713 Mabsadd L
.89619 .96476 2.37764 .77254 Mabsadd L
.89619 .96476 .56518 .7779 Mabsadd L
.89619 .96476 0 2.5 Mabsadd L closepath F
.89714 .96481 0 2.5 Mabsadd m
.89714 .96481 -0.56518 .7779 Mabsadd L
.89714 .96481 -2.37764 .77254 Mabsadd L
.89714 .96481 -0.91448 -0.29713 Mabsadd L
.89714 .96481 -1.46946 -2.02254 Mabsadd L
.89714 .96481 0 -0.96154 Mabsadd L
.89714 .96481 1.46946 -2.02254 Mabsadd L
.89714 .96481 .91448 -0.29713 Mabsadd L
.89714 .96481 2.37764 .77254 Mabsadd L
.89714 .96481 .56518 .7779 Mabsadd L
.89714 .96481 0 2.5 Mabsadd L closepath F
.8981 .96481 0 2.5 Mabsadd m
.8981 .96481 -0.56518 .7779 Mabsadd L
.8981 .96481 -2.37764 .77254 Mabsadd L
.8981 .96481 -0.91448 -0.29713 Mabsadd L
.8981 .96481 -1.46946 -2.02254 Mabsadd L
.8981 .96481 0 -0.96154 Mabsadd L
.8981 .96481 1.46946 -2.02254 Mabsadd L
.8981 .96481 .91448 -0.29713 Mabsadd L
.8981 .96481 2.37764 .77254 Mabsadd L
.8981 .96481 .56518 .7779 Mabsadd L
.8981 .96481 0 2.5 Mabsadd L closepath F
.89905 .96482 0 2.5 Mabsadd m
.89905 .96482 -0.56518 .7779 Mabsadd L
.89905 .96482 -2.37764 .77254 Mabsadd L
.89905 .96482 -0.91448 -0.29713 Mabsadd L
.89905 .96482 -1.46946 -2.02254 Mabsadd L
.89905 .96482 0 -0.96154 Mabsadd L
.89905 .96482 1.46946 -2.02254 Mabsadd L
.89905 .96482 .91448 -0.29713 Mabsadd L
.89905 .96482 2.37764 .77254 Mabsadd L
.89905 .96482 .56518 .7779 Mabsadd L
.89905 .96482 0 2.5 Mabsadd L closepath F
.9 .96483 0 2.5 Mabsadd m
.9 .96483 -0.56518 .7779 Mabsadd L
.9 .96483 -2.37764 .77254 Mabsadd L
.9 .96483 -0.91448 -0.29713 Mabsadd L
.9 .96483 -1.46946 -2.02254 Mabsadd L
.9 .96483 0 -0.96154 Mabsadd L
.9 .96483 1.46946 -2.02254 Mabsadd L
.9 .96483 .91448 -0.29713 Mabsadd L
.9 .96483 2.37764 .77254 Mabsadd L
.9 .96483 .56518 .7779 Mabsadd L
.9 .96483 0 2.5 Mabsadd L closepath F
.90095 .96483 0 2.5 Mabsadd m
.90095 .96483 -0.56518 .7779 Mabsadd L
.90095 .96483 -2.37764 .77254 Mabsadd L
.90095 .96483 -0.91448 -0.29713 Mabsadd L
.90095 .96483 -1.46946 -2.02254 Mabsadd L
.90095 .96483 0 -0.96154 Mabsadd L
.90095 .96483 1.46946 -2.02254 Mabsadd L
.90095 .96483 .91448 -0.29713 Mabsadd L
.90095 .96483 2.37764 .77254 Mabsadd L
.90095 .96483 .56518 .7779 Mabsadd L
.90095 .96483 0 2.5 Mabsadd L closepath F
.9019 .96483 0 2.5 Mabsadd m
.9019 .96483 -0.56518 .7779 Mabsadd L
.9019 .96483 -2.37764 .77254 Mabsadd L
.9019 .96483 -0.91448 -0.29713 Mabsadd L
.9019 .96483 -1.46946 -2.02254 Mabsadd L
.9019 .96483 0 -0.96154 Mabsadd L
.9019 .96483 1.46946 -2.02254 Mabsadd L
.9019 .96483 .91448 -0.29713 Mabsadd L
.9019 .96483 2.37764 .77254 Mabsadd L
.9019 .96483 .56518 .7779 Mabsadd L
.9019 .96483 0 2.5 Mabsadd L closepath F
.90286 .96483 0 2.5 Mabsadd m
.90286 .96483 -0.56518 .7779 Mabsadd L
.90286 .96483 -2.37764 .77254 Mabsadd L
.90286 .96483 -0.91448 -0.29713 Mabsadd L
.90286 .96483 -1.46946 -2.02254 Mabsadd L
.90286 .96483 0 -0.96154 Mabsadd L
.90286 .96483 1.46946 -2.02254 Mabsadd L
.90286 .96483 .91448 -0.29713 Mabsadd L
.90286 .96483 2.37764 .77254 Mabsadd L
.90286 .96483 .56518 .7779 Mabsadd L
.90286 .96483 0 2.5 Mabsadd L closepath F
.90381 .96484 0 2.5 Mabsadd m
.90381 .96484 -0.56518 .7779 Mabsadd L
.90381 .96484 -2.37764 .77254 Mabsadd L
.90381 .96484 -0.91448 -0.29713 Mabsadd L
.90381 .96484 -1.46946 -2.02254 Mabsadd L
.90381 .96484 0 -0.96154 Mabsadd L
.90381 .96484 1.46946 -2.02254 Mabsadd L
.90381 .96484 .91448 -0.29713 Mabsadd L
.90381 .96484 2.37764 .77254 Mabsadd L
.90381 .96484 .56518 .7779 Mabsadd L
.90381 .96484 0 2.5 Mabsadd L closepath F
.90476 .96484 0 2.5 Mabsadd m
.90476 .96484 -0.56518 .7779 Mabsadd L
.90476 .96484 -2.37764 .77254 Mabsadd L
.90476 .96484 -0.91448 -0.29713 Mabsadd L
.90476 .96484 -1.46946 -2.02254 Mabsadd L
.90476 .96484 0 -0.96154 Mabsadd L
.90476 .96484 1.46946 -2.02254 Mabsadd L
.90476 .96484 .91448 -0.29713 Mabsadd L
.90476 .96484 2.37764 .77254 Mabsadd L
.90476 .96484 .56518 .7779 Mabsadd L
.90476 .96484 0 2.5 Mabsadd L closepath F
.90571 .96484 0 2.5 Mabsadd m
.90571 .96484 -0.56518 .7779 Mabsadd L
.90571 .96484 -2.37764 .77254 Mabsadd L
.90571 .96484 -0.91448 -0.29713 Mabsadd L
.90571 .96484 -1.46946 -2.02254 Mabsadd L
.90571 .96484 0 -0.96154 Mabsadd L
.90571 .96484 1.46946 -2.02254 Mabsadd L
.90571 .96484 .91448 -0.29713 Mabsadd L
.90571 .96484 2.37764 .77254 Mabsadd L
.90571 .96484 .56518 .7779 Mabsadd L
.90571 .96484 0 2.5 Mabsadd L closepath F
.90667 .96484 0 2.5 Mabsadd m
.90667 .96484 -0.56518 .7779 Mabsadd L
.90667 .96484 -2.37764 .77254 Mabsadd L
.90667 .96484 -0.91448 -0.29713 Mabsadd L
.90667 .96484 -1.46946 -2.02254 Mabsadd L
.90667 .96484 0 -0.96154 Mabsadd L
.90667 .96484 1.46946 -2.02254 Mabsadd L
.90667 .96484 .91448 -0.29713 Mabsadd L
.90667 .96484 2.37764 .77254 Mabsadd L
.90667 .96484 .56518 .7779 Mabsadd L
.90667 .96484 0 2.5 Mabsadd L closepath F
.90762 .96484 0 2.5 Mabsadd m
.90762 .96484 -0.56518 .7779 Mabsadd L
.90762 .96484 -2.37764 .77254 Mabsadd L
.90762 .96484 -0.91448 -0.29713 Mabsadd L
.90762 .96484 -1.46946 -2.02254 Mabsadd L
.90762 .96484 0 -0.96154 Mabsadd L
.90762 .96484 1.46946 -2.02254 Mabsadd L
.90762 .96484 .91448 -0.29713 Mabsadd L
.90762 .96484 2.37764 .77254 Mabsadd L
.90762 .96484 .56518 .7779 Mabsadd L
.90762 .96484 0 2.5 Mabsadd L closepath F
.90857 .96484 0 2.5 Mabsadd m
.90857 .96484 -0.56518 .7779 Mabsadd L
.90857 .96484 -2.37764 .77254 Mabsadd L
.90857 .96484 -0.91448 -0.29713 Mabsadd L
.90857 .96484 -1.46946 -2.02254 Mabsadd L
.90857 .96484 0 -0.96154 Mabsadd L
.90857 .96484 1.46946 -2.02254 Mabsadd L
.90857 .96484 .91448 -0.29713 Mabsadd L
.90857 .96484 2.37764 .77254 Mabsadd L
.90857 .96484 .56518 .7779 Mabsadd L
.90857 .96484 0 2.5 Mabsadd L closepath F
.90952 .96484 0 2.5 Mabsadd m
.90952 .96484 -0.56518 .7779 Mabsadd L
.90952 .96484 -2.37764 .77254 Mabsadd L
.90952 .96484 -0.91448 -0.29713 Mabsadd L
.90952 .96484 -1.46946 -2.02254 Mabsadd L
.90952 .96484 0 -0.96154 Mabsadd L
.90952 .96484 1.46946 -2.02254 Mabsadd L
.90952 .96484 .91448 -0.29713 Mabsadd L
.90952 .96484 2.37764 .77254 Mabsadd L
.90952 .96484 .56518 .7779 Mabsadd L
.90952 .96484 0 2.5 Mabsadd L closepath F
.91048 .96483 0 2.5 Mabsadd m
.91048 .96483 -0.56518 .7779 Mabsadd L
.91048 .96483 -2.37764 .77254 Mabsadd L
.91048 .96483 -0.91448 -0.29713 Mabsadd L
.91048 .96483 -1.46946 -2.02254 Mabsadd L
.91048 .96483 0 -0.96154 Mabsadd L
.91048 .96483 1.46946 -2.02254 Mabsadd L
.91048 .96483 .91448 -0.29713 Mabsadd L
.91048 .96483 2.37764 .77254 Mabsadd L
.91048 .96483 .56518 .7779 Mabsadd L
.91048 .96483 0 2.5 Mabsadd L closepath F
.91143 .96483 0 2.5 Mabsadd m
.91143 .96483 -0.56518 .7779 Mabsadd L
.91143 .96483 -2.37764 .77254 Mabsadd L
.91143 .96483 -0.91448 -0.29713 Mabsadd L
.91143 .96483 -1.46946 -2.02254 Mabsadd L
.91143 .96483 0 -0.96154 Mabsadd L
.91143 .96483 1.46946 -2.02254 Mabsadd L
.91143 .96483 .91448 -0.29713 Mabsadd L
.91143 .96483 2.37764 .77254 Mabsadd L
.91143 .96483 .56518 .7779 Mabsadd L
.91143 .96483 0 2.5 Mabsadd L closepath F
.91238 .96483 0 2.5 Mabsadd m
.91238 .96483 -0.56518 .7779 Mabsadd L
.91238 .96483 -2.37764 .77254 Mabsadd L
.91238 .96483 -0.91448 -0.29713 Mabsadd L
.91238 .96483 -1.46946 -2.02254 Mabsadd L
.91238 .96483 0 -0.96154 Mabsadd L
.91238 .96483 1.46946 -2.02254 Mabsadd L
.91238 .96483 .91448 -0.29713 Mabsadd L
.91238 .96483 2.37764 .77254 Mabsadd L
.91238 .96483 .56518 .7779 Mabsadd L
.91238 .96483 0 2.5 Mabsadd L closepath F
.91333 .96482 0 2.5 Mabsadd m
.91333 .96482 -0.56518 .7779 Mabsadd L
.91333 .96482 -2.37764 .77254 Mabsadd L
.91333 .96482 -0.91448 -0.29713 Mabsadd L
.91333 .96482 -1.46946 -2.02254 Mabsadd L
.91333 .96482 0 -0.96154 Mabsadd L
.91333 .96482 1.46946 -2.02254 Mabsadd L
.91333 .96482 .91448 -0.29713 Mabsadd L
.91333 .96482 2.37764 .77254 Mabsadd L
.91333 .96482 .56518 .7779 Mabsadd L
.91333 .96482 0 2.5 Mabsadd L closepath F
.91429 .96481 0 2.5 Mabsadd m
.91429 .96481 -0.56518 .7779 Mabsadd L
.91429 .96481 -2.37764 .77254 Mabsadd L
.91429 .96481 -0.91448 -0.29713 Mabsadd L
.91429 .96481 -1.46946 -2.02254 Mabsadd L
.91429 .96481 0 -0.96154 Mabsadd L
.91429 .96481 1.46946 -2.02254 Mabsadd L
.91429 .96481 .91448 -0.29713 Mabsadd L
.91429 .96481 2.37764 .77254 Mabsadd L
.91429 .96481 .56518 .7779 Mabsadd L
.91429 .96481 0 2.5 Mabsadd L closepath F
.91524 .96481 0 2.5 Mabsadd m
.91524 .96481 -0.56518 .7779 Mabsadd L
.91524 .96481 -2.37764 .77254 Mabsadd L
.91524 .96481 -0.91448 -0.29713 Mabsadd L
.91524 .96481 -1.46946 -2.02254 Mabsadd L
.91524 .96481 0 -0.96154 Mabsadd L
.91524 .96481 1.46946 -2.02254 Mabsadd L
.91524 .96481 .91448 -0.29713 Mabsadd L
.91524 .96481 2.37764 .77254 Mabsadd L
.91524 .96481 .56518 .7779 Mabsadd L
.91524 .96481 0 2.5 Mabsadd L closepath F
.91619 .9648 0 2.5 Mabsadd m
.91619 .9648 -0.56518 .7779 Mabsadd L
.91619 .9648 -2.37764 .77254 Mabsadd L
.91619 .9648 -0.91448 -0.29713 Mabsadd L
.91619 .9648 -1.46946 -2.02254 Mabsadd L
.91619 .9648 0 -0.96154 Mabsadd L
.91619 .9648 1.46946 -2.02254 Mabsadd L
.91619 .9648 .91448 -0.29713 Mabsadd L
.91619 .9648 2.37764 .77254 Mabsadd L
.91619 .9648 .56518 .7779 Mabsadd L
.91619 .9648 0 2.5 Mabsadd L closepath F
.91714 .96476 0 2.5 Mabsadd m
.91714 .96476 -0.56518 .7779 Mabsadd L
.91714 .96476 -2.37764 .77254 Mabsadd L
.91714 .96476 -0.91448 -0.29713 Mabsadd L
.91714 .96476 -1.46946 -2.02254 Mabsadd L
.91714 .96476 0 -0.96154 Mabsadd L
.91714 .96476 1.46946 -2.02254 Mabsadd L
.91714 .96476 .91448 -0.29713 Mabsadd L
.91714 .96476 2.37764 .77254 Mabsadd L
.91714 .96476 .56518 .7779 Mabsadd L
.91714 .96476 0 2.5 Mabsadd L closepath F
.9181 .96475 0 2.5 Mabsadd m
.9181 .96475 -0.56518 .7779 Mabsadd L
.9181 .96475 -2.37764 .77254 Mabsadd L
.9181 .96475 -0.91448 -0.29713 Mabsadd L
.9181 .96475 -1.46946 -2.02254 Mabsadd L
.9181 .96475 0 -0.96154 Mabsadd L
.9181 .96475 1.46946 -2.02254 Mabsadd L
.9181 .96475 .91448 -0.29713 Mabsadd L
.9181 .96475 2.37764 .77254 Mabsadd L
.9181 .96475 .56518 .7779 Mabsadd L
.9181 .96475 0 2.5 Mabsadd L closepath F
.91905 .96469 0 2.5 Mabsadd m
.91905 .96469 -0.56518 .7779 Mabsadd L
.91905 .96469 -2.37764 .77254 Mabsadd L
.91905 .96469 -0.91448 -0.29713 Mabsadd L
.91905 .96469 -1.46946 -2.02254 Mabsadd L
.91905 .96469 0 -0.96154 Mabsadd L
.91905 .96469 1.46946 -2.02254 Mabsadd L
.91905 .96469 .91448 -0.29713 Mabsadd L
.91905 .96469 2.37764 .77254 Mabsadd L
.91905 .96469 .56518 .7779 Mabsadd L
.91905 .96469 0 2.5 Mabsadd L closepath F
.92 .96445 0 2.5 Mabsadd m
.92 .96445 -0.56518 .7779 Mabsadd L
.92 .96445 -2.37764 .77254 Mabsadd L
.92 .96445 -0.91448 -0.29713 Mabsadd L
.92 .96445 -1.46946 -2.02254 Mabsadd L
.92 .96445 0 -0.96154 Mabsadd L
.92 .96445 1.46946 -2.02254 Mabsadd L
.92 .96445 .91448 -0.29713 Mabsadd L
.92 .96445 2.37764 .77254 Mabsadd L
.92 .96445 .56518 .7779 Mabsadd L
.92 .96445 0 2.5 Mabsadd L closepath F
.92095 .96439 0 2.5 Mabsadd m
.92095 .96439 -0.56518 .7779 Mabsadd L
.92095 .96439 -2.37764 .77254 Mabsadd L
.92095 .96439 -0.91448 -0.29713 Mabsadd L
.92095 .96439 -1.46946 -2.02254 Mabsadd L
.92095 .96439 0 -0.96154 Mabsadd L
.92095 .96439 1.46946 -2.02254 Mabsadd L
.92095 .96439 .91448 -0.29713 Mabsadd L
.92095 .96439 2.37764 .77254 Mabsadd L
.92095 .96439 .56518 .7779 Mabsadd L
.92095 .96439 0 2.5 Mabsadd L closepath F
.9219 .96387 0 2.5 Mabsadd m
.9219 .96387 -0.56518 .7779 Mabsadd L
.9219 .96387 -2.37764 .77254 Mabsadd L
.9219 .96387 -0.91448 -0.29713 Mabsadd L
.9219 .96387 -1.46946 -2.02254 Mabsadd L
.9219 .96387 0 -0.96154 Mabsadd L
.9219 .96387 1.46946 -2.02254 Mabsadd L
.9219 .96387 .91448 -0.29713 Mabsadd L
.9219 .96387 2.37764 .77254 Mabsadd L
.9219 .96387 .56518 .7779 Mabsadd L
.9219 .96387 0 2.5 Mabsadd L closepath F
.92286 .96358 0 2.5 Mabsadd m
.92286 .96358 -0.56518 .7779 Mabsadd L
.92286 .96358 -2.37764 .77254 Mabsadd L
.92286 .96358 -0.91448 -0.29713 Mabsadd L
.92286 .96358 -1.46946 -2.02254 Mabsadd L
.92286 .96358 0 -0.96154 Mabsadd L
.92286 .96358 1.46946 -2.02254 Mabsadd L
.92286 .96358 .91448 -0.29713 Mabsadd L
.92286 .96358 2.37764 .77254 Mabsadd L
.92286 .96358 .56518 .7779 Mabsadd L
.92286 .96358 0 2.5 Mabsadd L closepath F
.92381 .96051 0 2.5 Mabsadd m
.92381 .96051 -0.56518 .7779 Mabsadd L
.92381 .96051 -2.37764 .77254 Mabsadd L
.92381 .96051 -0.91448 -0.29713 Mabsadd L
.92381 .96051 -1.46946 -2.02254 Mabsadd L
.92381 .96051 0 -0.96154 Mabsadd L
.92381 .96051 1.46946 -2.02254 Mabsadd L
.92381 .96051 .91448 -0.29713 Mabsadd L
.92381 .96051 2.37764 .77254 Mabsadd L
.92381 .96051 .56518 .7779 Mabsadd L
.92381 .96051 0 2.5 Mabsadd L closepath F
.92476 .95842 0 2.5 Mabsadd m
.92476 .95842 -0.56518 .7779 Mabsadd L
.92476 .95842 -2.37764 .77254 Mabsadd L
.92476 .95842 -0.91448 -0.29713 Mabsadd L
.92476 .95842 -1.46946 -2.02254 Mabsadd L
.92476 .95842 0 -0.96154 Mabsadd L
.92476 .95842 1.46946 -2.02254 Mabsadd L
.92476 .95842 .91448 -0.29713 Mabsadd L
.92476 .95842 2.37764 .77254 Mabsadd L
.92476 .95842 .56518 .7779 Mabsadd L
.92476 .95842 0 2.5 Mabsadd L closepath F
.92571 .93346 0 2.5 Mabsadd m
.92571 .93346 -0.56518 .7779 Mabsadd L
.92571 .93346 -2.37764 .77254 Mabsadd L
.92571 .93346 -0.91448 -0.29713 Mabsadd L
.92571 .93346 -1.46946 -2.02254 Mabsadd L
.92571 .93346 0 -0.96154 Mabsadd L
.92571 .93346 1.46946 -2.02254 Mabsadd L
.92571 .93346 .91448 -0.29713 Mabsadd L
.92571 .93346 2.37764 .77254 Mabsadd L
.92571 .93346 .56518 .7779 Mabsadd L
.92571 .93346 0 2.5 Mabsadd L closepath F
.92667 .91125 0 2.5 Mabsadd m
.92667 .91125 -0.56518 .7779 Mabsadd L
.92667 .91125 -2.37764 .77254 Mabsadd L
.92667 .91125 -0.91448 -0.29713 Mabsadd L
.92667 .91125 -1.46946 -2.02254 Mabsadd L
.92667 .91125 0 -0.96154 Mabsadd L
.92667 .91125 1.46946 -2.02254 Mabsadd L
.92667 .91125 .91448 -0.29713 Mabsadd L
.92667 .91125 2.37764 .77254 Mabsadd L
.92667 .91125 .56518 .7779 Mabsadd L
.92667 .91125 0 2.5 Mabsadd L closepath F
.92762 .91126 0 2.5 Mabsadd m
.92762 .91126 -0.56518 .7779 Mabsadd L
.92762 .91126 -2.37764 .77254 Mabsadd L
.92762 .91126 -0.91448 -0.29713 Mabsadd L
.92762 .91126 -1.46946 -2.02254 Mabsadd L
.92762 .91126 0 -0.96154 Mabsadd L
.92762 .91126 1.46946 -2.02254 Mabsadd L
.92762 .91126 .91448 -0.29713 Mabsadd L
.92762 .91126 2.37764 .77254 Mabsadd L
.92762 .91126 .56518 .7779 Mabsadd L
.92762 .91126 0 2.5 Mabsadd L closepath F
.92857 .91189 0 2.5 Mabsadd m
.92857 .91189 -0.56518 .7779 Mabsadd L
.92857 .91189 -2.37764 .77254 Mabsadd L
.92857 .91189 -0.91448 -0.29713 Mabsadd L
.92857 .91189 -1.46946 -2.02254 Mabsadd L
.92857 .91189 0 -0.96154 Mabsadd L
.92857 .91189 1.46946 -2.02254 Mabsadd L
.92857 .91189 .91448 -0.29713 Mabsadd L
.92857 .91189 2.37764 .77254 Mabsadd L
.92857 .91189 .56518 .7779 Mabsadd L
.92857 .91189 0 2.5 Mabsadd L closepath F
.92952 .92438 0 2.5 Mabsadd m
.92952 .92438 -0.56518 .7779 Mabsadd L
.92952 .92438 -2.37764 .77254 Mabsadd L
.92952 .92438 -0.91448 -0.29713 Mabsadd L
.92952 .92438 -1.46946 -2.02254 Mabsadd L
.92952 .92438 0 -0.96154 Mabsadd L
.92952 .92438 1.46946 -2.02254 Mabsadd L
.92952 .92438 .91448 -0.29713 Mabsadd L
.92952 .92438 2.37764 .77254 Mabsadd L
.92952 .92438 .56518 .7779 Mabsadd L
.92952 .92438 0 2.5 Mabsadd L closepath F
.93048 .93846 0 2.5 Mabsadd m
.93048 .93846 -0.56518 .7779 Mabsadd L
.93048 .93846 -2.37764 .77254 Mabsadd L
.93048 .93846 -0.91448 -0.29713 Mabsadd L
.93048 .93846 -1.46946 -2.02254 Mabsadd L
.93048 .93846 0 -0.96154 Mabsadd L
.93048 .93846 1.46946 -2.02254 Mabsadd L
.93048 .93846 .91448 -0.29713 Mabsadd L
.93048 .93846 2.37764 .77254 Mabsadd L
.93048 .93846 .56518 .7779 Mabsadd L
.93048 .93846 0 2.5 Mabsadd L closepath F
.93143 .93855 0 2.5 Mabsadd m
.93143 .93855 -0.56518 .7779 Mabsadd L
.93143 .93855 -2.37764 .77254 Mabsadd L
.93143 .93855 -0.91448 -0.29713 Mabsadd L
.93143 .93855 -1.46946 -2.02254 Mabsadd L
.93143 .93855 0 -0.96154 Mabsadd L
.93143 .93855 1.46946 -2.02254 Mabsadd L
.93143 .93855 .91448 -0.29713 Mabsadd L
.93143 .93855 2.37764 .77254 Mabsadd L
.93143 .93855 .56518 .7779 Mabsadd L
.93143 .93855 0 2.5 Mabsadd L closepath F
.93238 .93984 0 2.5 Mabsadd m
.93238 .93984 -0.56518 .7779 Mabsadd L
.93238 .93984 -2.37764 .77254 Mabsadd L
.93238 .93984 -0.91448 -0.29713 Mabsadd L
.93238 .93984 -1.46946 -2.02254 Mabsadd L
.93238 .93984 0 -0.96154 Mabsadd L
.93238 .93984 1.46946 -2.02254 Mabsadd L
.93238 .93984 .91448 -0.29713 Mabsadd L
.93238 .93984 2.37764 .77254 Mabsadd L
.93238 .93984 .56518 .7779 Mabsadd L
.93238 .93984 0 2.5 Mabsadd L closepath F
.93333 .94393 0 2.5 Mabsadd m
.93333 .94393 -0.56518 .7779 Mabsadd L
.93333 .94393 -2.37764 .77254 Mabsadd L
.93333 .94393 -0.91448 -0.29713 Mabsadd L
.93333 .94393 -1.46946 -2.02254 Mabsadd L
.93333 .94393 0 -0.96154 Mabsadd L
.93333 .94393 1.46946 -2.02254 Mabsadd L
.93333 .94393 .91448 -0.29713 Mabsadd L
.93333 .94393 2.37764 .77254 Mabsadd L
.93333 .94393 .56518 .7779 Mabsadd L
.93333 .94393 0 2.5 Mabsadd L closepath F
.93429 .94466 0 2.5 Mabsadd m
.93429 .94466 -0.56518 .7779 Mabsadd L
.93429 .94466 -2.37764 .77254 Mabsadd L
.93429 .94466 -0.91448 -0.29713 Mabsadd L
.93429 .94466 -1.46946 -2.02254 Mabsadd L
.93429 .94466 0 -0.96154 Mabsadd L
.93429 .94466 1.46946 -2.02254 Mabsadd L
.93429 .94466 .91448 -0.29713 Mabsadd L
.93429 .94466 2.37764 .77254 Mabsadd L
.93429 .94466 .56518 .7779 Mabsadd L
.93429 .94466 0 2.5 Mabsadd L closepath F
.93524 .94446 0 2.5 Mabsadd m
.93524 .94446 -0.56518 .7779 Mabsadd L
.93524 .94446 -2.37764 .77254 Mabsadd L
.93524 .94446 -0.91448 -0.29713 Mabsadd L
.93524 .94446 -1.46946 -2.02254 Mabsadd L
.93524 .94446 0 -0.96154 Mabsadd L
.93524 .94446 1.46946 -2.02254 Mabsadd L
.93524 .94446 .91448 -0.29713 Mabsadd L
.93524 .94446 2.37764 .77254 Mabsadd L
.93524 .94446 .56518 .7779 Mabsadd L
.93524 .94446 0 2.5 Mabsadd L closepath F
.93619 .94158 0 2.5 Mabsadd m
.93619 .94158 -0.56518 .7779 Mabsadd L
.93619 .94158 -2.37764 .77254 Mabsadd L
.93619 .94158 -0.91448 -0.29713 Mabsadd L
.93619 .94158 -1.46946 -2.02254 Mabsadd L
.93619 .94158 0 -0.96154 Mabsadd L
.93619 .94158 1.46946 -2.02254 Mabsadd L
.93619 .94158 .91448 -0.29713 Mabsadd L
.93619 .94158 2.37764 .77254 Mabsadd L
.93619 .94158 .56518 .7779 Mabsadd L
.93619 .94158 0 2.5 Mabsadd L closepath F
.93714 .94461 0 2.5 Mabsadd m
.93714 .94461 -0.56518 .7779 Mabsadd L
.93714 .94461 -2.37764 .77254 Mabsadd L
.93714 .94461 -0.91448 -0.29713 Mabsadd L
.93714 .94461 -1.46946 -2.02254 Mabsadd L
.93714 .94461 0 -0.96154 Mabsadd L
.93714 .94461 1.46946 -2.02254 Mabsadd L
.93714 .94461 .91448 -0.29713 Mabsadd L
.93714 .94461 2.37764 .77254 Mabsadd L
.93714 .94461 .56518 .7779 Mabsadd L
.93714 .94461 0 2.5 Mabsadd L closepath F
.9381 .9539 0 2.5 Mabsadd m
.9381 .9539 -0.56518 .7779 Mabsadd L
.9381 .9539 -2.37764 .77254 Mabsadd L
.9381 .9539 -0.91448 -0.29713 Mabsadd L
.9381 .9539 -1.46946 -2.02254 Mabsadd L
.9381 .9539 0 -0.96154 Mabsadd L
.9381 .9539 1.46946 -2.02254 Mabsadd L
.9381 .9539 .91448 -0.29713 Mabsadd L
.9381 .9539 2.37764 .77254 Mabsadd L
.9381 .9539 .56518 .7779 Mabsadd L
.9381 .9539 0 2.5 Mabsadd L closepath F
.93905 .95393 0 2.5 Mabsadd m
.93905 .95393 -0.56518 .7779 Mabsadd L
.93905 .95393 -2.37764 .77254 Mabsadd L
.93905 .95393 -0.91448 -0.29713 Mabsadd L
.93905 .95393 -1.46946 -2.02254 Mabsadd L
.93905 .95393 0 -0.96154 Mabsadd L
.93905 .95393 1.46946 -2.02254 Mabsadd L
.93905 .95393 .91448 -0.29713 Mabsadd L
.93905 .95393 2.37764 .77254 Mabsadd L
.93905 .95393 .56518 .7779 Mabsadd L
.93905 .95393 0 2.5 Mabsadd L closepath F
.94 .95399 0 2.5 Mabsadd m
.94 .95399 -0.56518 .7779 Mabsadd L
.94 .95399 -2.37764 .77254 Mabsadd L
.94 .95399 -0.91448 -0.29713 Mabsadd L
.94 .95399 -1.46946 -2.02254 Mabsadd L
.94 .95399 0 -0.96154 Mabsadd L
.94 .95399 1.46946 -2.02254 Mabsadd L
.94 .95399 .91448 -0.29713 Mabsadd L
.94 .95399 2.37764 .77254 Mabsadd L
.94 .95399 .56518 .7779 Mabsadd L
.94 .95399 0 2.5 Mabsadd L closepath F
.94095 .95401 0 2.5 Mabsadd m
.94095 .95401 -0.56518 .7779 Mabsadd L
.94095 .95401 -2.37764 .77254 Mabsadd L
.94095 .95401 -0.91448 -0.29713 Mabsadd L
.94095 .95401 -1.46946 -2.02254 Mabsadd L
.94095 .95401 0 -0.96154 Mabsadd L
.94095 .95401 1.46946 -2.02254 Mabsadd L
.94095 .95401 .91448 -0.29713 Mabsadd L
.94095 .95401 2.37764 .77254 Mabsadd L
.94095 .95401 .56518 .7779 Mabsadd L
.94095 .95401 0 2.5 Mabsadd L closepath F
.9419 .95397 0 2.5 Mabsadd m
.9419 .95397 -0.56518 .7779 Mabsadd L
.9419 .95397 -2.37764 .77254 Mabsadd L
.9419 .95397 -0.91448 -0.29713 Mabsadd L
.9419 .95397 -1.46946 -2.02254 Mabsadd L
.9419 .95397 0 -0.96154 Mabsadd L
.9419 .95397 1.46946 -2.02254 Mabsadd L
.9419 .95397 .91448 -0.29713 Mabsadd L
.9419 .95397 2.37764 .77254 Mabsadd L
.9419 .95397 .56518 .7779 Mabsadd L
.9419 .95397 0 2.5 Mabsadd L closepath F
.94286 .9539 0 2.5 Mabsadd m
.94286 .9539 -0.56518 .7779 Mabsadd L
.94286 .9539 -2.37764 .77254 Mabsadd L
.94286 .9539 -0.91448 -0.29713 Mabsadd L
.94286 .9539 -1.46946 -2.02254 Mabsadd L
.94286 .9539 0 -0.96154 Mabsadd L
.94286 .9539 1.46946 -2.02254 Mabsadd L
.94286 .9539 .91448 -0.29713 Mabsadd L
.94286 .9539 2.37764 .77254 Mabsadd L
.94286 .9539 .56518 .7779 Mabsadd L
.94286 .9539 0 2.5 Mabsadd L closepath F
.94381 .95382 0 2.5 Mabsadd m
.94381 .95382 -0.56518 .7779 Mabsadd L
.94381 .95382 -2.37764 .77254 Mabsadd L
.94381 .95382 -0.91448 -0.29713 Mabsadd L
.94381 .95382 -1.46946 -2.02254 Mabsadd L
.94381 .95382 0 -0.96154 Mabsadd L
.94381 .95382 1.46946 -2.02254 Mabsadd L
.94381 .95382 .91448 -0.29713 Mabsadd L
.94381 .95382 2.37764 .77254 Mabsadd L
.94381 .95382 .56518 .7779 Mabsadd L
.94381 .95382 0 2.5 Mabsadd L closepath F
.94476 .95399 0 2.5 Mabsadd m
.94476 .95399 -0.56518 .7779 Mabsadd L
.94476 .95399 -2.37764 .77254 Mabsadd L
.94476 .95399 -0.91448 -0.29713 Mabsadd L
.94476 .95399 -1.46946 -2.02254 Mabsadd L
.94476 .95399 0 -0.96154 Mabsadd L
.94476 .95399 1.46946 -2.02254 Mabsadd L
.94476 .95399 .91448 -0.29713 Mabsadd L
.94476 .95399 2.37764 .77254 Mabsadd L
.94476 .95399 .56518 .7779 Mabsadd L
.94476 .95399 0 2.5 Mabsadd L closepath F
.94571 .95456 0 2.5 Mabsadd m
.94571 .95456 -0.56518 .7779 Mabsadd L
.94571 .95456 -2.37764 .77254 Mabsadd L
.94571 .95456 -0.91448 -0.29713 Mabsadd L
.94571 .95456 -1.46946 -2.02254 Mabsadd L
.94571 .95456 0 -0.96154 Mabsadd L
.94571 .95456 1.46946 -2.02254 Mabsadd L
.94571 .95456 .91448 -0.29713 Mabsadd L
.94571 .95456 2.37764 .77254 Mabsadd L
.94571 .95456 .56518 .7779 Mabsadd L
.94571 .95456 0 2.5 Mabsadd L closepath F
.94667 .95459 0 2.5 Mabsadd m
.94667 .95459 -0.56518 .7779 Mabsadd L
.94667 .95459 -2.37764 .77254 Mabsadd L
.94667 .95459 -0.91448 -0.29713 Mabsadd L
.94667 .95459 -1.46946 -2.02254 Mabsadd L
.94667 .95459 0 -0.96154 Mabsadd L
.94667 .95459 1.46946 -2.02254 Mabsadd L
.94667 .95459 .91448 -0.29713 Mabsadd L
.94667 .95459 2.37764 .77254 Mabsadd L
.94667 .95459 .56518 .7779 Mabsadd L
.94667 .95459 0 2.5 Mabsadd L closepath F
.94762 .95472 0 2.5 Mabsadd m
.94762 .95472 -0.56518 .7779 Mabsadd L
.94762 .95472 -2.37764 .77254 Mabsadd L
.94762 .95472 -0.91448 -0.29713 Mabsadd L
.94762 .95472 -1.46946 -2.02254 Mabsadd L
.94762 .95472 0 -0.96154 Mabsadd L
.94762 .95472 1.46946 -2.02254 Mabsadd L
.94762 .95472 .91448 -0.29713 Mabsadd L
.94762 .95472 2.37764 .77254 Mabsadd L
.94762 .95472 .56518 .7779 Mabsadd L
.94762 .95472 0 2.5 Mabsadd L closepath F
.94857 .95473 0 2.5 Mabsadd m
.94857 .95473 -0.56518 .7779 Mabsadd L
.94857 .95473 -2.37764 .77254 Mabsadd L
.94857 .95473 -0.91448 -0.29713 Mabsadd L
.94857 .95473 -1.46946 -2.02254 Mabsadd L
.94857 .95473 0 -0.96154 Mabsadd L
.94857 .95473 1.46946 -2.02254 Mabsadd L
.94857 .95473 .91448 -0.29713 Mabsadd L
.94857 .95473 2.37764 .77254 Mabsadd L
.94857 .95473 .56518 .7779 Mabsadd L
.94857 .95473 0 2.5 Mabsadd L closepath F
.94952 .95476 0 2.5 Mabsadd m
.94952 .95476 -0.56518 .7779 Mabsadd L
.94952 .95476 -2.37764 .77254 Mabsadd L
.94952 .95476 -0.91448 -0.29713 Mabsadd L
.94952 .95476 -1.46946 -2.02254 Mabsadd L
.94952 .95476 0 -0.96154 Mabsadd L
.94952 .95476 1.46946 -2.02254 Mabsadd L
.94952 .95476 .91448 -0.29713 Mabsadd L
.94952 .95476 2.37764 .77254 Mabsadd L
.94952 .95476 .56518 .7779 Mabsadd L
.94952 .95476 0 2.5 Mabsadd L closepath F
.95048 .95482 0 2.5 Mabsadd m
.95048 .95482 -0.56518 .7779 Mabsadd L
.95048 .95482 -2.37764 .77254 Mabsadd L
.95048 .95482 -0.91448 -0.29713 Mabsadd L
.95048 .95482 -1.46946 -2.02254 Mabsadd L
.95048 .95482 0 -0.96154 Mabsadd L
.95048 .95482 1.46946 -2.02254 Mabsadd L
.95048 .95482 .91448 -0.29713 Mabsadd L
.95048 .95482 2.37764 .77254 Mabsadd L
.95048 .95482 .56518 .7779 Mabsadd L
.95048 .95482 0 2.5 Mabsadd L closepath F
.95143 .95482 0 2.5 Mabsadd m
.95143 .95482 -0.56518 .7779 Mabsadd L
.95143 .95482 -2.37764 .77254 Mabsadd L
.95143 .95482 -0.91448 -0.29713 Mabsadd L
.95143 .95482 -1.46946 -2.02254 Mabsadd L
.95143 .95482 0 -0.96154 Mabsadd L
.95143 .95482 1.46946 -2.02254 Mabsadd L
.95143 .95482 .91448 -0.29713 Mabsadd L
.95143 .95482 2.37764 .77254 Mabsadd L
.95143 .95482 .56518 .7779 Mabsadd L
.95143 .95482 0 2.5 Mabsadd L closepath F
.95238 .95483 0 2.5 Mabsadd m
.95238 .95483 -0.56518 .7779 Mabsadd L
.95238 .95483 -2.37764 .77254 Mabsadd L
.95238 .95483 -0.91448 -0.29713 Mabsadd L
.95238 .95483 -1.46946 -2.02254 Mabsadd L
.95238 .95483 0 -0.96154 Mabsadd L
.95238 .95483 1.46946 -2.02254 Mabsadd L
.95238 .95483 .91448 -0.29713 Mabsadd L
.95238 .95483 2.37764 .77254 Mabsadd L
.95238 .95483 .56518 .7779 Mabsadd L
.95238 .95483 0 2.5 Mabsadd L closepath F
.95333 .95484 0 2.5 Mabsadd m
.95333 .95484 -0.56518 .7779 Mabsadd L
.95333 .95484 -2.37764 .77254 Mabsadd L
.95333 .95484 -0.91448 -0.29713 Mabsadd L
.95333 .95484 -1.46946 -2.02254 Mabsadd L
.95333 .95484 0 -0.96154 Mabsadd L
.95333 .95484 1.46946 -2.02254 Mabsadd L
.95333 .95484 .91448 -0.29713 Mabsadd L
.95333 .95484 2.37764 .77254 Mabsadd L
.95333 .95484 .56518 .7779 Mabsadd L
.95333 .95484 0 2.5 Mabsadd L closepath F
.95429 .95484 0 2.5 Mabsadd m
.95429 .95484 -0.56518 .7779 Mabsadd L
.95429 .95484 -2.37764 .77254 Mabsadd L
.95429 .95484 -0.91448 -0.29713 Mabsadd L
.95429 .95484 -1.46946 -2.02254 Mabsadd L
.95429 .95484 0 -0.96154 Mabsadd L
.95429 .95484 1.46946 -2.02254 Mabsadd L
.95429 .95484 .91448 -0.29713 Mabsadd L
.95429 .95484 2.37764 .77254 Mabsadd L
.95429 .95484 .56518 .7779 Mabsadd L
.95429 .95484 0 2.5 Mabsadd L closepath F
.95524 .95484 0 2.5 Mabsadd m
.95524 .95484 -0.56518 .7779 Mabsadd L
.95524 .95484 -2.37764 .77254 Mabsadd L
.95524 .95484 -0.91448 -0.29713 Mabsadd L
.95524 .95484 -1.46946 -2.02254 Mabsadd L
.95524 .95484 0 -0.96154 Mabsadd L
.95524 .95484 1.46946 -2.02254 Mabsadd L
.95524 .95484 .91448 -0.29713 Mabsadd L
.95524 .95484 2.37764 .77254 Mabsadd L
.95524 .95484 .56518 .7779 Mabsadd L
.95524 .95484 0 2.5 Mabsadd L closepath F
.95619 .95485 0 2.5 Mabsadd m
.95619 .95485 -0.56518 .7779 Mabsadd L
.95619 .95485 -2.37764 .77254 Mabsadd L
.95619 .95485 -0.91448 -0.29713 Mabsadd L
.95619 .95485 -1.46946 -2.02254 Mabsadd L
.95619 .95485 0 -0.96154 Mabsadd L
.95619 .95485 1.46946 -2.02254 Mabsadd L
.95619 .95485 .91448 -0.29713 Mabsadd L
.95619 .95485 2.37764 .77254 Mabsadd L
.95619 .95485 .56518 .7779 Mabsadd L
.95619 .95485 0 2.5 Mabsadd L closepath F
.95714 .95485 0 2.5 Mabsadd m
.95714 .95485 -0.56518 .7779 Mabsadd L
.95714 .95485 -2.37764 .77254 Mabsadd L
.95714 .95485 -0.91448 -0.29713 Mabsadd L
.95714 .95485 -1.46946 -2.02254 Mabsadd L
.95714 .95485 0 -0.96154 Mabsadd L
.95714 .95485 1.46946 -2.02254 Mabsadd L
.95714 .95485 .91448 -0.29713 Mabsadd L
.95714 .95485 2.37764 .77254 Mabsadd L
.95714 .95485 .56518 .7779 Mabsadd L
.95714 .95485 0 2.5 Mabsadd L closepath F
.9581 .95485 0 2.5 Mabsadd m
.9581 .95485 -0.56518 .7779 Mabsadd L
.9581 .95485 -2.37764 .77254 Mabsadd L
.9581 .95485 -0.91448 -0.29713 Mabsadd L
.9581 .95485 -1.46946 -2.02254 Mabsadd L
.9581 .95485 0 -0.96154 Mabsadd L
.9581 .95485 1.46946 -2.02254 Mabsadd L
.9581 .95485 .91448 -0.29713 Mabsadd L
.9581 .95485 2.37764 .77254 Mabsadd L
.9581 .95485 .56518 .7779 Mabsadd L
.9581 .95485 0 2.5 Mabsadd L closepath F
.95905 .95485 0 2.5 Mabsadd m
.95905 .95485 -0.56518 .7779 Mabsadd L
.95905 .95485 -2.37764 .77254 Mabsadd L
.95905 .95485 -0.91448 -0.29713 Mabsadd L
.95905 .95485 -1.46946 -2.02254 Mabsadd L
.95905 .95485 0 -0.96154 Mabsadd L
.95905 .95485 1.46946 -2.02254 Mabsadd L
.95905 .95485 .91448 -0.29713 Mabsadd L
.95905 .95485 2.37764 .77254 Mabsadd L
.95905 .95485 .56518 .7779 Mabsadd L
.95905 .95485 0 2.5 Mabsadd L closepath F
.96 .95485 0 2.5 Mabsadd m
.96 .95485 -0.56518 .7779 Mabsadd L
.96 .95485 -2.37764 .77254 Mabsadd L
.96 .95485 -0.91448 -0.29713 Mabsadd L
.96 .95485 -1.46946 -2.02254 Mabsadd L
.96 .95485 0 -0.96154 Mabsadd L
.96 .95485 1.46946 -2.02254 Mabsadd L
.96 .95485 .91448 -0.29713 Mabsadd L
.96 .95485 2.37764 .77254 Mabsadd L
.96 .95485 .56518 .7779 Mabsadd L
.96 .95485 0 2.5 Mabsadd L closepath F
.96095 .95485 0 2.5 Mabsadd m
.96095 .95485 -0.56518 .7779 Mabsadd L
.96095 .95485 -2.37764 .77254 Mabsadd L
.96095 .95485 -0.91448 -0.29713 Mabsadd L
.96095 .95485 -1.46946 -2.02254 Mabsadd L
.96095 .95485 0 -0.96154 Mabsadd L
.96095 .95485 1.46946 -2.02254 Mabsadd L
.96095 .95485 .91448 -0.29713 Mabsadd L
.96095 .95485 2.37764 .77254 Mabsadd L
.96095 .95485 .56518 .7779 Mabsadd L
.96095 .95485 0 2.5 Mabsadd L closepath F
.9619 .95485 0 2.5 Mabsadd m
.9619 .95485 -0.56518 .7779 Mabsadd L
.9619 .95485 -2.37764 .77254 Mabsadd L
.9619 .95485 -0.91448 -0.29713 Mabsadd L
.9619 .95485 -1.46946 -2.02254 Mabsadd L
.9619 .95485 0 -0.96154 Mabsadd L
.9619 .95485 1.46946 -2.02254 Mabsadd L
.9619 .95485 .91448 -0.29713 Mabsadd L
.9619 .95485 2.37764 .77254 Mabsadd L
.9619 .95485 .56518 .7779 Mabsadd L
.9619 .95485 0 2.5 Mabsadd L closepath F
.96286 .95485 0 2.5 Mabsadd m
.96286 .95485 -0.56518 .7779 Mabsadd L
.96286 .95485 -2.37764 .77254 Mabsadd L
.96286 .95485 -0.91448 -0.29713 Mabsadd L
.96286 .95485 -1.46946 -2.02254 Mabsadd L
.96286 .95485 0 -0.96154 Mabsadd L
.96286 .95485 1.46946 -2.02254 Mabsadd L
.96286 .95485 .91448 -0.29713 Mabsadd L
.96286 .95485 2.37764 .77254 Mabsadd L
.96286 .95485 .56518 .7779 Mabsadd L
.96286 .95485 0 2.5 Mabsadd L closepath F
.96381 .95484 0 2.5 Mabsadd m
.96381 .95484 -0.56518 .7779 Mabsadd L
.96381 .95484 -2.37764 .77254 Mabsadd L
.96381 .95484 -0.91448 -0.29713 Mabsadd L
.96381 .95484 -1.46946 -2.02254 Mabsadd L
.96381 .95484 0 -0.96154 Mabsadd L
.96381 .95484 1.46946 -2.02254 Mabsadd L
.96381 .95484 .91448 -0.29713 Mabsadd L
.96381 .95484 2.37764 .77254 Mabsadd L
.96381 .95484 .56518 .7779 Mabsadd L
.96381 .95484 0 2.5 Mabsadd L closepath F
.96476 .95484 0 2.5 Mabsadd m
.96476 .95484 -0.56518 .7779 Mabsadd L
.96476 .95484 -2.37764 .77254 Mabsadd L
.96476 .95484 -0.91448 -0.29713 Mabsadd L
.96476 .95484 -1.46946 -2.02254 Mabsadd L
.96476 .95484 0 -0.96154 Mabsadd L
.96476 .95484 1.46946 -2.02254 Mabsadd L
.96476 .95484 .91448 -0.29713 Mabsadd L
.96476 .95484 2.37764 .77254 Mabsadd L
.96476 .95484 .56518 .7779 Mabsadd L
.96476 .95484 0 2.5 Mabsadd L closepath F
.96571 .95484 0 2.5 Mabsadd m
.96571 .95484 -0.56518 .7779 Mabsadd L
.96571 .95484 -2.37764 .77254 Mabsadd L
.96571 .95484 -0.91448 -0.29713 Mabsadd L
.96571 .95484 -1.46946 -2.02254 Mabsadd L
.96571 .95484 0 -0.96154 Mabsadd L
.96571 .95484 1.46946 -2.02254 Mabsadd L
.96571 .95484 .91448 -0.29713 Mabsadd L
.96571 .95484 2.37764 .77254 Mabsadd L
.96571 .95484 .56518 .7779 Mabsadd L
.96571 .95484 0 2.5 Mabsadd L closepath F
.96667 .95483 0 2.5 Mabsadd m
.96667 .95483 -0.56518 .7779 Mabsadd L
.96667 .95483 -2.37764 .77254 Mabsadd L
.96667 .95483 -0.91448 -0.29713 Mabsadd L
.96667 .95483 -1.46946 -2.02254 Mabsadd L
.96667 .95483 0 -0.96154 Mabsadd L
.96667 .95483 1.46946 -2.02254 Mabsadd L
.96667 .95483 .91448 -0.29713 Mabsadd L
.96667 .95483 2.37764 .77254 Mabsadd L
.96667 .95483 .56518 .7779 Mabsadd L
.96667 .95483 0 2.5 Mabsadd L closepath F
.96762 .95482 0 2.5 Mabsadd m
.96762 .95482 -0.56518 .7779 Mabsadd L
.96762 .95482 -2.37764 .77254 Mabsadd L
.96762 .95482 -0.91448 -0.29713 Mabsadd L
.96762 .95482 -1.46946 -2.02254 Mabsadd L
.96762 .95482 0 -0.96154 Mabsadd L
.96762 .95482 1.46946 -2.02254 Mabsadd L
.96762 .95482 .91448 -0.29713 Mabsadd L
.96762 .95482 2.37764 .77254 Mabsadd L
.96762 .95482 .56518 .7779 Mabsadd L
.96762 .95482 0 2.5 Mabsadd L closepath F
.96857 .95481 0 2.5 Mabsadd m
.96857 .95481 -0.56518 .7779 Mabsadd L
.96857 .95481 -2.37764 .77254 Mabsadd L
.96857 .95481 -0.91448 -0.29713 Mabsadd L
.96857 .95481 -1.46946 -2.02254 Mabsadd L
.96857 .95481 0 -0.96154 Mabsadd L
.96857 .95481 1.46946 -2.02254 Mabsadd L
.96857 .95481 .91448 -0.29713 Mabsadd L
.96857 .95481 2.37764 .77254 Mabsadd L
.96857 .95481 .56518 .7779 Mabsadd L
.96857 .95481 0 2.5 Mabsadd L closepath F
.96952 .9548 0 2.5 Mabsadd m
.96952 .9548 -0.56518 .7779 Mabsadd L
.96952 .9548 -2.37764 .77254 Mabsadd L
.96952 .9548 -0.91448 -0.29713 Mabsadd L
.96952 .9548 -1.46946 -2.02254 Mabsadd L
.96952 .9548 0 -0.96154 Mabsadd L
.96952 .9548 1.46946 -2.02254 Mabsadd L
.96952 .9548 .91448 -0.29713 Mabsadd L
.96952 .9548 2.37764 .77254 Mabsadd L
.96952 .9548 .56518 .7779 Mabsadd L
.96952 .9548 0 2.5 Mabsadd L closepath F
.97048 .95477 0 2.5 Mabsadd m
.97048 .95477 -0.56518 .7779 Mabsadd L
.97048 .95477 -2.37764 .77254 Mabsadd L
.97048 .95477 -0.91448 -0.29713 Mabsadd L
.97048 .95477 -1.46946 -2.02254 Mabsadd L
.97048 .95477 0 -0.96154 Mabsadd L
.97048 .95477 1.46946 -2.02254 Mabsadd L
.97048 .95477 .91448 -0.29713 Mabsadd L
.97048 .95477 2.37764 .77254 Mabsadd L
.97048 .95477 .56518 .7779 Mabsadd L
.97048 .95477 0 2.5 Mabsadd L closepath F
.97143 .95476 0 2.5 Mabsadd m
.97143 .95476 -0.56518 .7779 Mabsadd L
.97143 .95476 -2.37764 .77254 Mabsadd L
.97143 .95476 -0.91448 -0.29713 Mabsadd L
.97143 .95476 -1.46946 -2.02254 Mabsadd L
.97143 .95476 0 -0.96154 Mabsadd L
.97143 .95476 1.46946 -2.02254 Mabsadd L
.97143 .95476 .91448 -0.29713 Mabsadd L
.97143 .95476 2.37764 .77254 Mabsadd L
.97143 .95476 .56518 .7779 Mabsadd L
.97143 .95476 0 2.5 Mabsadd L closepath F
.97238 .9547 0 2.5 Mabsadd m
.97238 .9547 -0.56518 .7779 Mabsadd L
.97238 .9547 -2.37764 .77254 Mabsadd L
.97238 .9547 -0.91448 -0.29713 Mabsadd L
.97238 .9547 -1.46946 -2.02254 Mabsadd L
.97238 .9547 0 -0.96154 Mabsadd L
.97238 .9547 1.46946 -2.02254 Mabsadd L
.97238 .9547 .91448 -0.29713 Mabsadd L
.97238 .9547 2.37764 .77254 Mabsadd L
.97238 .9547 .56518 .7779 Mabsadd L
.97238 .9547 0 2.5 Mabsadd L closepath F
.97333 .95448 0 2.5 Mabsadd m
.97333 .95448 -0.56518 .7779 Mabsadd L
.97333 .95448 -2.37764 .77254 Mabsadd L
.97333 .95448 -0.91448 -0.29713 Mabsadd L
.97333 .95448 -1.46946 -2.02254 Mabsadd L
.97333 .95448 0 -0.96154 Mabsadd L
.97333 .95448 1.46946 -2.02254 Mabsadd L
.97333 .95448 .91448 -0.29713 Mabsadd L
.97333 .95448 2.37764 .77254 Mabsadd L
.97333 .95448 .56518 .7779 Mabsadd L
.97333 .95448 0 2.5 Mabsadd L closepath F
.97429 .95443 0 2.5 Mabsadd m
.97429 .95443 -0.56518 .7779 Mabsadd L
.97429 .95443 -2.37764 .77254 Mabsadd L
.97429 .95443 -0.91448 -0.29713 Mabsadd L
.97429 .95443 -1.46946 -2.02254 Mabsadd L
.97429 .95443 0 -0.96154 Mabsadd L
.97429 .95443 1.46946 -2.02254 Mabsadd L
.97429 .95443 .91448 -0.29713 Mabsadd L
.97429 .95443 2.37764 .77254 Mabsadd L
.97429 .95443 .56518 .7779 Mabsadd L
.97429 .95443 0 2.5 Mabsadd L closepath F
.97524 .95396 0 2.5 Mabsadd m
.97524 .95396 -0.56518 .7779 Mabsadd L
.97524 .95396 -2.37764 .77254 Mabsadd L
.97524 .95396 -0.91448 -0.29713 Mabsadd L
.97524 .95396 -1.46946 -2.02254 Mabsadd L
.97524 .95396 0 -0.96154 Mabsadd L
.97524 .95396 1.46946 -2.02254 Mabsadd L
.97524 .95396 .91448 -0.29713 Mabsadd L
.97524 .95396 2.37764 .77254 Mabsadd L
.97524 .95396 .56518 .7779 Mabsadd L
.97524 .95396 0 2.5 Mabsadd L closepath F
.97619 .9537 0 2.5 Mabsadd m
.97619 .9537 -0.56518 .7779 Mabsadd L
.97619 .9537 -2.37764 .77254 Mabsadd L
.97619 .9537 -0.91448 -0.29713 Mabsadd L
.97619 .9537 -1.46946 -2.02254 Mabsadd L
.97619 .9537 0 -0.96154 Mabsadd L
.97619 .9537 1.46946 -2.02254 Mabsadd L
.97619 .9537 .91448 -0.29713 Mabsadd L
.97619 .9537 2.37764 .77254 Mabsadd L
.97619 .9537 .56518 .7779 Mabsadd L
.97619 .9537 0 2.5 Mabsadd L closepath F
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[12]:=",
  ImageSize->{288, 288},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181000`40n007h00Ooooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00
oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00
>?oo0P009_oo10000ooo0P009Ooo00<00?ooool00_oo0P009?oo0P001?oo0P009?oo0P001?oo0P00
8Ooo10000ooo0P001?oo0P000ooo000gool01000ooooo`009Ooo00<00?ooool00ooo00@00?ooool0
02Coo`05003oooooool00002ool00`00ooooo`0Pool01000ooooo`000_oo00@00?ooool002;oo`04
003ooooo0002ool01000ooooo`008_oo00D00?ooooooo`0000;oo`04003ooooo0002ool10001ool1
ool003Ooo`04003ooooo000Vool00`00ooooo`02ool01000ooooo`008Ooo1@000_oo00@00?ooool0
02;oo`04003ooooo0002ool01000ooooo`008_oo00@00?ooool000;oo`04003ooooo000Rool01@00
oooooooo00000_oo00@00?ooool000;oo`40007oo`7oo`00=ooo00@00?ooool002Ooo`05003ooooo
ool00002ool00`00ooooo`0Oool01000ooooo`000ooo00@00?ooool002;oo`<000?oo`04003ooooo
000Sool20003ool01000ooooo`008_oo00D00?ooooooo`0000;oo`04003ooooo0002ool10001ool1
ool003Ooo`04003ooooo000Xool01000ooooo`000_oo00<00?ooool08?oo00<00?oo00000ooo00@0
0?ooool002;oo`03003ooooo00?oo`04003ooooo000Rool01000ooooo`000_oo00@00?ooool002;o
o`05003oooooool00002ool01000ooooo`000_oo0@000Ooo0Ooo000gool01000ooooo`009Ooo00@0
0?ooool000;oo`04003ooooo000Sool20003ool01000ooooo`008_oo00<00?ooool00ooo00@00?oo
ool002;oo`04003ooooo0002ool01000ooooo`008?oo0`000ooo00@00?ooool000;oo`04003ooooo
0002ool003Soo`8002Ooo`8000Coo`8002Goo`03003ooooo00;oo`8002Coo`<000?oo`8002Coo`80
00Coo`8002?oo`03003ooooo00;oo`8000Coo`8000?oo`00oooo8Ooo003ooolQool00?ooob7oo`00
=?ooj0001?oo000dool00`00ooooo`02ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08
ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08
ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08
ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08ool00`00ooooo`08
ool00`00ooooo`08ool00`00ooooo`09ool00`00ooooo`02ool00`00ooooo`02ool003Coo`03003o
oooo00;oo`03003ooooo02Woo`03003ooooo02Woo`03003ooooo02Woo`03003ooooo02Woo`03003o
oooo02[oo`03003ooooo00;oo`03003ooooo00;oo`00=?oo00<00?ooool0i?oo00<00?ooool00_oo
000]ool20005ool00`00ooooo`02ool00oP0ooooo`3Oool00`00ooooo`02ool002coo`04003ooooo
0004ool01000oooooooo100OAooo0P0O??oo0_P02ooo00?h0?ooool0B_oo00<00?ooool00_oo000/
ool01000ooooo`001?oo0P000_oo100O>?oo1@0O2_oo0P0O<Ooo3_P02Ooo0oP0B_oo0P001?oo000/
ool01000ooooo`001?oo00<00?ooool01P0O1_oo0`0O2_oo0`0O6?oo3`0O1ooo1@0O;ooo3oP02?oo
1?P0Booo00<00?ooool00_oo000/ool01000ooooo`001?oo00@00?ooooooo`D07`Koo`<07`[oo`80
7aSooa007`Ooo`D07bkoo`oh00[oo`;h04coo`03003ooooo00;oo`00;?oo00@00?ooool000Coo`03
003ooooo00Coo`0301oooooo00Coo`D07`Ooo`H07aKoo`l07`Ooo`D07`[oo`;h02Coo`kh05Soo`03
003ooooo00;oo`00;Ooo0P001Ooo00<00?ooool03Ooo00<07oooool02Ooo0P0O2_oo0`0O2_oo4P0O
1Ooo1@0O2Ooo1?P08_oo1?P01ooo1?P0F?oo00<00?ooool00_oo000dool00`00ooooo`0=ool201l9
ool501l8ool301l9ool501l8ool201l:ool301l;ool3n00Rool4n007ool4n009ool3n01<ool00`00
ooooo`02ool003Coo`03003ooooo00?oo`807`Ooo`@07`Woo`807`Woo`D07`Soo`H07`Goo`D07`Wo
o`807`goo`03n03ooooo02;oo`;h00Soo`Gh00Ooo`Ch04coo`03003ooooo00;oo`00=?oo00<00?oo
ool00_oo1@0O1Ooo0`0O2_oo100O2?oo100O1ooo1@0O5?oo100O6?oo00?h0?ooool05_oo0_P01ooo
1_P02?oo0oP0C?oo00<00?ooool00_oo000dool00`00ooooo`03ool301l6ool201l9ool701l6ool6
01l6ool501lEool201lHool3n00Fool3n008ool4n00:ool2n01<ool00`00ooooo`02ool003Coo`03
003ooooo00;oo`D07`Coo`D07`Woo`D07`Ooo`D07`Ooo`807`goo`D07`?oo`D07aKoo`Ch01Ooo`;h
00Soo`Gh00Ooo`Gh04_oo`03003ooooo00;oo`00=?oo0P000ooo3`0O2?oo1P0O1?oo200O1Ooo1@0O
2ooo3@0O2_oo0P0O2ooo0_P06Ooo00?h0?ooool01ooo1?P01_oo1_P0B_oo0P001?oo000dool00`00
ooooo`02ool@01l6oolC01l7ool00`0Oooooo`0:ool>01l:ool301l_ool4n007ool4n01<ool00`00
ooooo`02ool003Coo`03003ooooo00?ooa007`Cooa807`Soo`807`coo`<07`03ool07`0O00H07`[o
o`H07booo`;h00Soo`Ch04coo`03003ooooo00;oo`00=?oo00<00?ooool01ooo100O1?oo900O2ooo
1@0O4ooo1@0O<?oo3OP0C?oo00<00?ooool00_oo000dool00`00ooooo`0>ool=01l3ool801l00ooo
01l07`0;01l:ool501l9ool201l9ool501l^ool>n01<ool00`00ooooo`02ool003Coo`03003ooooo
013oo`T07`goo`h07`Woo`D07`Woo`D07`Woo`0301oooooo02ooo`kh04_oo`03003ooooo00;oo`00
=?oo00<00?ooool0:_oo200O2ooo1@0O2ooo0P0O>_oo0_P000?oooP0n0002oP0Booo00<00?ooool0
0_oo000dool00`00ooooo`0nool201l=ool201l7ool00`0Oool07`0_ool4n00:ool2n01<ool00`00
ooooo`02ool003Coo`03003ooooo04_oo`D07`Koo`807c7oo`?h00_oo`;h04_oo`03003ooooo00;o
o`00=?oo00<00?ooool0C?oo1@0O1?oo1@0O<?oo00?h0?ooool02Ooo1?P0B_oo00<00?ooool00_oo
000dool00`00ooooo`1>ool201l6ool201lGool3n00Tool4n008ool00oP0ooooo`0oool00`00oooo
o`02ool003Coo`8000ooo`;h03coo`h07aCoo`Gh02Coo`;h00Soo`?h03ooo`8000Coo`00=?oo00<0
0?ooool03Ooo1?P0>ooo00?h000O01l02P0O5_oo1?P0;Ooo1?P0@?oo00<00?ooool00_oo000dool0
0`00ooooo`0>ool3n00jool2n00<01lGool3n00]ool5n00oool00`00ooooo`02ool003Coo`03003o
oooo00ooo`03n03ooooo03[oo`;h00;oo`H07`ooo`;h00Soo`Ch02Coo`kh00[oo`03n03ooooo03;o
o`03003ooooo00;oo`00=?oo00<00?ooool0COoo00?h0?ooool05?oo1?P02?oo0_P09?oo3_P02_oo
0oP0<ooo00<00?ooool00_oo000dool00`00ooooo`1Uool3n007ool2n00Vool<n00<ool2n00cool0
0`00ooooo`02ool003Coo`03003ooooo05Woo`807`[oo`?h00Koo`Ch02Koo`[h00goo`03n03ooooo
03;oo`03003ooooo00;oo`00=?oo00<00?ooool0FOoo00<07oooool02?oo3oP0LOoo00<00?ooool0
0_oo000dool00`00ooooo`1Gool501l9ool>n01aool00`00ooooo`02ool003Coo`03003ooooo06Ko
o`ch04Woo`;h02Ooo`03003ooooo00;oo`00=?oo0P00F_oo0P0O3?oo2OP0B_oo1?P09Ooo0P001?oo
000dool00`00ooooo`1Iool301l7ool301mEool3n00Vool00`00ooooo`02ool003Coo`03003ooooo
05Ooo`D07`Ooo`807eOoo`03n03ooooo02Goo`03003ooooo00;oo`00=?oo00<00?ooool0F?oo1@0O
1Ooo1@0OOOoo00<00?ooool00_oo000dool00`00ooooo`1Hool501l7ool00`0Oooooo`1mool00`00
ooooo`02ool003Coo`03003ooooo05Soo`Ch00Soo`807gkoo`03003ooooo00;oo`00=?oo00<00?oo
ool0F?oo0_P00P0O1ooo00?h000On000O_oo00<00?ooool00_oo000dool00`00ooooo`1Iool00oP0
01l07`06ool501m6ool3n00dool00`00ooooo`02ool003Coo`03003ooooo00;oo`;h05Coo`D07`Ko
o`807`03n03ooooo04Coo`Ch03Coo`03003ooooo00;oo`00=?oo00@00?ooooooo`Ch05Goo`807`Go
o`D07dOoo`?h03Coo`03003ooooo00;oo`00=?oo0P000ooo0oP05Ooo0_P08ooo0_P05ooo00?h000O
01l02`0OB?oo0_P0<ooo0P001?oo000dool01000oooooooo0oP05Ooo1?P08Ooo1?P05Ooo0oP0300O
?Ooo00?h0?ooool02Ooo00?h0?ooool0<_oo00<00?ooool00_oo000dool00`00ooooo`02ool2n00F
ool3n00Rool3n00Fool00oP001l07`0=01ljool3n009ool3n00cool00`00ooooo`02ool003Coo`03
003ooooo00;oo`?h01Koo`03n03ooooo02;oo`03n03ooooo01Goo`Ch00`07cWoo`Ch00Soo`Ch03?o
o`03003ooooo00;oo`00=?oo00@00?ooooooo`Ch05?oo`Gh00Koo`;h000301oooooo03[oo`?h00Wo
o`;h03Coo`03003ooooo00;oo`00=?oo00<00?ooool00_oo1?P0DOoo3oP0?Ooo00?h0?ooool0?_oo
00<00?ooool00_oo000dool00`00ooooo`02ool4n01@oolAn00kool3n00Rool2n00Kool00`00oooo
o`02ool003Coo`03003ooooo00?oo`?h00Ooo`;h04Oooa7h03[oo`Ch027oo`Ch01[oo`03003ooooo
00;oo`003Ooo0`000ooo00D00?ooooooo`0000Koo`@000?oo`8000goo`03003ooooo00Coo`03n03o
oooo00Koo`?h04Ooo`;h0003oooh0?P000_h03_oo`Ch00Soo`03n03ooooo01Koo`?h01[oo`03003o
oooo00;oo`004?oo00L00?ooooooo`00ool00009ool01@00oooooooo00000_oo00<00?ooool02_oo
00<00?ooool03?oo1?P0E?oo0_P0?Ooo0oP01ooo0oP06?oo00?h0?ooool06Ooo00<00?ooool00_oo
000@ool00`00ooooo`02ool00`00ooooo`08ool01@00oooooooo00000_oo00<00?ooool02_oo00<0
0?ooool03Ooo0_P0Tooo1?P01_oo1?P0=?oo00<00?ooool00_oo000=ool30004ool00`00ool00009
ool01@00oooooooo00000_oo00<00?ooool02_oo0P003_oo0oP0??oo0_P0EOoo1?P01_oo0_P0=?oo
0P001?oo000=ool00`00ooooo`03ool01@00oooooooo00002?oo00D00?ooooooo`0000;oo`03003o
oooo00[oo`03003ooooo00coo`Ch03_oo`Ch01Ooo`03n03ooooo00Koo`<07c7oo`gh03Coo`03003o
oooo00;oo`003Ooo00<00?ooool03_oo0`000ooo00@00?ooool000Ooo`<000;oo`<000?oo`;h00So
o`;h03_oo`Gh01Koo`?h00Ooo`<07c;oo`ch03Coo`03003ooooo00;oo`003Ooo10003ooo00<00?oo
ool00_oo0P002Ooo00@00?ooooooo`8000?oo`Ch00Koo`Ch03_oo`Gh00Ooo`03n03ooooo00[oo`04
n0007`0On006ool501lbool:n00eool00`00ooooo`02ool002[oo`<000?oo`04003oooooool20004
ool3n005ool5n00kool5n006ool3n00<ool201l9ool00`0Oooooo`0bool9n0000ooon03h0009ool2
n00Xool00`00ooooo`02ool0033oo`04003oooooool20004ool=n00`ool00oP0ooooo`08ool5n005
ool4n00;ool401l8ool201llool4n007ool4n00Wool00`00ooooo`02ool002ooo`8000?oo`8000Co
o`gh02ooo`?h00Soo`oh00coo`;h00Woo`<07c_oo`Ch00Ooo`Ch02Ooo`03003ooooo00;oo`00<?oo
00@00?ooooooo`8000Goo`_h02ooo`Ch00Oooa3h00coo`03n03ooooo00Ooo`@07c[oo`Kh00Ooo`;h
02Soo`03003ooooo00;oo`00=?oo00<00?ooool01Ooo2OP0<Ooo0_P02Ooo3oP03?oo0`0O1ooo100O
>ooo1OP01ooo0_P0:?oo00<00?ooool00_oo000dool00`00ooooo`0oool3n009ool2n002ool9n00=
ool201l7ool401llool5n006ool4n00Wool00`00ooooo`02ool003Coo`8001[oo`;h02?oo`Ch02;o
o`l07c_oo`Ch00Soo`?h02Koo`8000Coo`00=?oo00<00?ooool06?oo1?P08_oo1?P09?oo300O>ooo
1?P02_oo00?h0?ooool05_oo00?h0?ooool03Ooo00<00?ooool00_oo000dool00`00ooooo`0Hool4
n00Qool6n00Qool>01llool3n00Sool2n00>ool00`00ooooo`02ool003Coo`03003ooooo00koo`;h
00Woo`Ch01Ooo`03n03ooooo00Ooo`Gh02Coo`0301oooooo04Goo`?h00Ooo`;h01Soo`?h00koo`03
003ooooo00;oo`00=?oo00<00?ooool03Ooo1?P02?oo1?P02_oo00?h0?ooool02Ooo0oP02Ooo1?P0
K?oo1?P01Ooo1?P06?oo00?h0?ooool03Ooo00<00?ooool00_oo000dool00`00ooooo`0>ool3n009
ool2n00:ool3n009ool4n009ool4n01]ool3n006ool3n00Xool00`00ooooo`02ool003Coo`03003o
oooo00ooo`03n03ooooo00Ooo`?h00Woo`Ch00Woo`Ch00[oo`;h06koo`ch02Soo`03003ooooo00;o
o`00=?oo00<00?ooool03_oo0_P02?oo1?P02_oo0_P02_oo0oP05_oo00?h0?ooool0;ooo0_P0<Ooo
3?P0:?oo00<00?ooool00_oo000dool00`00ooooo`0=ool4n008ool3n00Gool00oP0ooooo`0Dool3
n00_ool4n00aool<n00Wool00`00ooooo`02ool003Coo`03003ooooo00koo`Ch027oo`03n03ooooo
01?oo`Ch033oo`?h03;oo`_h02Ooo`03003ooooo00;oo`00=?oo00<00?ooool03_oo3?P06?oo0oP0
3?oo0_P01ooo0_P0<_oo00?h0?ooool0>_oo0_P09ooo00<00?ooool00_oo000dool2000@ool<n00F
ool5n00:ool=n01]ool3n00Vool20004ool003Coo`03003ooooo00koo`gh01Koo`Gh00Woo`kh06ko
o`?h02Koo`03003ooooo00;oo`00=?oo00<00?ooool03ooo2oP05ooo1?P02ooo3?P0K_oo1?P02Ooo
00?h0?ooool06_oo00<00?ooool00_oo000dool00`00ooooo`0Kool00oP0ooooo`0Dool3n00=ool9
n01`ool3n008ool3n00Kool00`00ooooo`02ool003Coo`03003ooooo01_oo`;h00Woo`;h00_oo`03
n03oooP008Koo`;h00Ooo`Ch01_oo`03003ooooo00;oo`00=?oo00<00?ooool06_oo0oP02?oo1?P0
2ooo0_P02?oo0_P0O?oo0oP01ooo0_P07?oo00<00?ooool00_oo000dool00`00ooooo`0Kool00oP0
ooooo`07ool4n00:ool4n006ool4n01jool4n008ool2n00Kool00`00ooooo`02ool003Coo`03003o
oooo01_oo`03n03ooooo00Koo`Ch00[oo`Gh00Ooo`?h07[oo`Ch00Ooo`?h01_oo`03003ooooo00;o
o`00=?oo00<00?ooool06_oo0oP02?oo0oP02ooo0oP02?oo0_P0Nooo1?P02?oo00?h0?ooool06_oo
00<00?ooool00_oo000dool00`00ooooo`0Iool4n008ool4n00Dool3n01lool2n009ool00oP0oooo
o`0Jool00`00ooooo`02ool003Coo`8001_oo`Ch00Soo`?h00_oo`_h08Ooo`;h01_oo`8000Coo`00
=?oo00<00?ooool06_oo1?P01ooo1?P02_oo3OP0QOoo1?P06ooo00<00?ooool00_oo000dool00`00
ooooo`0Kool>n00;ool<n026ool4n00Jool00`00ooooo`02ool003Coo`03003ooooo01[oo`kh00go
o`[h08Ooo`Ch01[oo`03003ooooo00;oo`00=?oo00<00?ooool06ooo3?P000?oooP0ool0Dooo00?h
0?ooool0?Ooo00?h0?ooool01ooo0_P02_oo00?h0?ooool03_oo00<00?ooool00_oo000dool00`00
ooooo`0Lool:n0001?oon03h0?P0D_oo0oP0?Ooo0oP01ooo00?h0?ooool02Ooo0oP03ooo00<00?oo
ool00_oo000dool00`00ooooo`0Vool4n017ool2n008ool5n00mool;n009ool4n00?ool00`00oooo
o`02ool003Coo`03003ooooo02Koo`?h04Ooo`Ch00Soo`Ch03coo`ch00[oo`?h00ooo`03003ooooo
00;oo`00=?oo00<00?ooool09ooo0_P02Ooo00?h0?ooool0??oo0oP01ooo1?P0?Ooo3?P02ooo0oP0
3_oo00<00?ooool00_oo000dool00`00ooooo`0Xool00oP0ooooo`06ool3n00nool2n008ool3n00n
ool:n0000ooon03oo`08ool4n00>ool00`00ooooo`02ool003Coo`8002Woo`03n03ooooo00Ooo`;h
03goo`Ch00Koo`Ch04Soo`?h00Soo`Ch00goo`8000Coo`00=?oo00<00?ooool09ooo0oP02?oo00?h
0?ooool0??oo3_P0BOoo0_P02?oo0oP03ooo00<00?ooool00_oo000dool00`00ooooo`0Vool4n007
ool3n00mool=n01:ool00oP0ooooo`06ool5n00>ool00`00ooooo`02ool003Coo`03003ooooo02Oo
o`?h00Koo`Ch03goo`gh04Woo`?h00Soo`Ch00koo`03003ooooo00;oo`00=?oo00<00?ooool0:?oo
0oP01_oo0_P0?ooo2oP0BOoo1?P02_oo00?h0?ooool03Ooo00<00?ooool00_oo000dool00`00oooo
o`0Xool<n00lool301mDool4n007ool2n00@ool00`00ooooo`02ool003Coo`03003ooooo02Ooo`gh
03coo`807eGoo`gh013oo`03003ooooo00;oo`003Ooo10000_oo00D00?ooooooo`0000Koo`@000?o
o`8000goo`03003ooooo02Soo`_h03coo`D07`[oo`03n03ooooo04Koo`gh013oo`03003ooooo00;o
o`003ooo00<00?ooool00_oo00<00?oo00002Ooo00D00?ooooooo`0000;oo`03003ooooo00[oo`03
003ooooo02[oo`Sh03goo`D07`Woo`?h04Soo`ch013oo`03003ooooo00;oo`003ooo00<00?ooool0
0ooo00<00?ooool02?oo00D00?ooooooo`0000;oo`03003ooooo00[oo`03003ooooo073oo`@07`So
o`Ch04Woo`[h017oo`03003ooooo00;oo`003ooo00<00?ooool00_oo00<00?oo00002Ooo00D00?oo
ooooo`0000;oo`03003ooooo00[oo`03003ooooo07goo`;h05Goo`;h00koo`03003ooooo00;oo`00
3ooo00D00?ooooooo`0000?oo`03003ooooo00Koo`05003oooooool00002ool00`00ooooo`0:ool2
001oool00oP0ooooo`1Bool4n00<ool20004ool000goo`<000koo`<000?oo`04003ooooo000<ool0
0`00ooooo`1mool3n01Dool3n00=ool00`00ooooo`02ool000ooo`03003ooooo00koo`03003ooooo
00;oo`8000Soo`<000;oo`8007;oo`807`Woo`Ch05Goo`03n03ooooo00coo`03003ooooo00;oo`00
<?oo00L00?ooooooo`00ool0001`ool501l8ool00oP001l07`1Eool00oP0ooooo`0<ool00`00oooo
o`02ool002[oo`<000?oo`07003oooooool00?oo0000L?oo0_P000<07oooool01ooo0`0OEOoo0_P0
3_oo00<00?ooool00_oo000`ool01`00oooooooo003oo`00077oo`;h00Ooo`L07e;oo`?h00koo`03
003ooooo00;oo`00;ooo0P000ooo00<00?oo0000Nooo100OE?oo1?P03Ooo00<00?ooool00_oo000`
ool01000oooooooo0P00OOoo0P0OE_oo3?P01?oo00<00?ooool00_oo000dool00`00ooooo`1bool2
01l7ool501mEool;n004ool00`00ooooo`02ool003Coo`03003ooooo07;oo`807`Ooo`@07`Woo`<0
7dWoo`ch00Coo`03003ooooo00;oo`00=?oo0P00LOoo1@0O1Ooo1P0O2?oo0P0OBooo2oP00ooo0P00
1?oo000dool00`00ooooo`1bool>01l7ool501mHool00`00ooooo`02ool003Coo`03003ooooo073o
oa407`Ooo`<07eWoo`03003ooooo00;oo`00=?oo00<00?ooool0Looo3P0O1_oo1@0OF?oo00<00?oo
ool00_oo000dool00`00ooooo`1mool401l6ool501mHool00`00ooooo`02ool003Coo`03003ooooo
07koo`l07eOoo`03003ooooo00;oo`00=?oo00<00?ooool0OOoo400OEooo00<00?ooool00_oo000d
ool00`00ooooo`1mool=01mJool00`00ooooo`02ool003Coo`03003ooooo0>Coo`03003ooooo00;o
o`00=?oo00<00?ooool0i?oo00<00?ooool00_oo000dool2003Tool20004ool003Coo`03003ooooo
0>Coo`03003ooooo00;oo`00=?oo00<00?ooool0i?oo00<00?ooool00_oo000dool00`00ooooo`3T
ool00`00ooooo`02ool003Coo`03003ooooo08[oo`<07`Ooo`0301ooo`0O04goo`03003ooooo00;o
o`00=?oo00<00?ooool0R_oo0`0O1ooo0`0OCOoo00<00?ooool00_oo000dool00`00ooooo`29ool4
01l6ool501m<ool00`00ooooo`02ool003Coo`03003ooooo08Woo`D07`Ooo`<07dcoo`03003ooooo
00;oo`00=?oo00<00?ooool0Rooo00<07oooool01Ooo1@0OC?oo00<00?ooool00_oo000dool00`00
ooooo`2;ool401l4ool601m;ool00`00ooooo`02ool003Coo`03003ooooo08_oo`l07d[oo`03003o
oooo00;oo`00=?oo0P00R_oo3`0OBooo0P001?oo000dool00`00ooooo`2<ool901m?ool00`00oooo
o`02ool003Coo`03003ooooo077oo`03n03ooooo073oo`03003ooooo00;oo`00=?oo00<00?ooool0
L?oo0oP0LOoo00<00?ooool00_oo000dool00`00ooooo`1_ool4n01aool00`00ooooo`02ool003Co
o`03003ooooo073oo`;h07;oo`03003ooooo00;oo`00=?oo00<00?ooool0i?oo00<00?ooool00_oo
000dool00`00ooooo`3Tool00`00ooooo`02ool003Coo`03003ooooo0>Coo`03003ooooo00;oo`00
=?oo00<00?ooool0i?oo00<00?ooool00_oo000dool2002Gool301m:ool20004ool003Coo`03003o
oooo09Koo`807dcoo`03003ooooo00;oo`00=?oo00<00?ooool0UOoo1@0OB_oo00<00?ooool00_oo
000dool00`00ooooo`2Eool501l7ool201m1ool00`00ooooo`02ool003Coo`03003ooooo077oo`;h
02koo`<07d3oo`03003ooooo00;oo`00=?oo00<00?ooool0L?oo1?P0:ooo1@0O@?oo00<00?ooool0
0_oo000dool00`00ooooo`1aool3n00/ool501loool00`00ooooo`02ool00003ool00000008000?o
o`03003ooooo00;oo`<000?oo`05003oooooool00006ool40003ool2000=ool00`00ooooo`1bool0
0oP0ooooo`0Rool401l5ool301m1ool00`00ooooo`02ool000?oo`03003ooooo00[oo`07003ooooo
ool00?oo00002Ooo00D00?ooooooo`0000;oo`03003ooooo00[oo`03003ooooo077oo`;h02Coo`d0
7d3oo`03003ooooo00;oo`000ooo00<00?ooool02_oo00<00?ooool00_oo00<00?ooool02?oo00D0
0?ooooooo`0000;oo`03003ooooo00[oo`03003ooooo073oo`Ch027oo`h07d7oo`03003ooooo00;o
o`000ooo00<00?ooool01ooo0`001?oo00<00?oo00002Ooo00D00?ooooooo`0000;oo`03003ooooo
00[oo`03003ooooo077oo`?h02Coo`X07d;oo`03003ooooo00;oo`000ooo00<00?ooool01ooo00<0
0?ooool00ooo00D00?ooooooo`0000Soo`05003oooooool00002ool00`00ooooo`0:ool2001cool0
0oP0ooooo`1^ool20004ool00004ool000000009ool00`00ooooo`0>ool30003ool01000ooooo`00
3?oo00<00?ooool0i?oo00<00?ooool00_oo0003ool00`00ooooo`07ool4000?ool00`00ooooo`02
ool20008ool30002ool2003Uool00`00ooooo`02ool0033oo`07003oooooool00?oo0000i?oo00<0
0?ooool00_oo000Zool30003ool01`00oooooooo003oo`000:7oo`<07d3oo`03003ooooo00;oo`00
<?oo00L00?ooooooo`00ool0002Qool201m1ool00`00ooooo`02ool002ooo`8000?oo`03003oo`00
0:3oo`D07cooo`03003ooooo00;oo`00<?oo00@00?ooooooo`800:?oo`0301oooooo03ooo`03003o
oooo00;oo`00=?oo00<00?ooool0i?oo00<00?ooool00_oo000dool00`00ooooo`3Tool00`00oooo
o`02ool003Coo`800>Coo`8000Coo`00=?oo00<00?ooool0i?oo00<00?ooool00_oo000dool00`00
ooooo`3Tool00`00ooooo`02ool003Coo`03003ooooo0>Coo`03003ooooo00;oo`00=?oo00<00?oo
ool0i?oo00<00?ooool00_oo000dool00`00ooooo`3Tool00`00ooooo`02ool003Coo`03003ooooo
0:;oo`807d3oo`03003ooooo00;oo`00=?oo00<00?ooool0X_oo0`0O?ooo00<00?ooool00_oo000d
ool00`00ooooo`2Pool501loool00`00ooooo`02ool003Coo`03003ooooo0:7oo`D07ckoo`03003o
oooo00;oo`00=?oo0P00Y?oo00<07oooool0?Ooo0P001?oo000dool00`00ooooo`3Tool00`00oooo
o`02ool003Coo`03003ooooo0>Coo`03003ooooo00;oo`00=?oo00<00?ooool0i?oo00<00?ooool0
0_oo000dool00`00ooooo`3Tool00`00ooooo`02ool003Coo`03003ooooo0:goo`<07cCoo`03003o
oooo00;oo`00=?oo00<00?ooool0[Ooo0P0O=Ooo00<00?ooool00_oo000dool00`00ooooo`2/ool5
01lcool00`00ooooo`02ool003Coo`03003ooooo0:koo`0301oooooo03?oo`03003ooooo00;oo`00
=?oo00<00?ooool0i?oo00<00?ooool00_oo000dool00`00ooooo`3Tool00`00ooooo`02ool003Co
o`800>Coo`8000Coo`00=?oo00<00?ooool0Xooo0P0O2Ooo0P0O=?oo00<00?ooool00_oo000dool0
0`00ooooo`2Rool401l8ool201ldool00`00ooooo`02ool003Coo`03003ooooo0:7oo`D07`Koo`D0
7c?oo`03003ooooo00;oo`00=?oo00<00?ooool0Xooo0`0O1Ooo1@0O=?oo00<00?ooool00_oo000d
ool00`00ooooo`2Rool501l6ool201leool00`00ooooo`02ool003Coo`03003ooooo0:;oo`D07`Ko
o`807cGoo`03003ooooo00;oo`00=?oo00<00?ooool0Xooo3@0O=?oo00<00?ooool00_oo000dool0
0`00ooooo`2Tool>01lbool00`00ooooo`02ool003Coo`03003ooooo0:?oo`h07`Soo`<07bSoo`03
003ooooo00;oo`00=?oo0P00Yooo1`0O00?oo`0O01l00P0O1ooo0`0O9ooo0P001?oo000dool00`00
ooooo`2^ool401l6ool501lWool00`00ooooo`02ool003Coo`03003ooooo0:goo`D07`Soo`0301oo
oooo02Ooo`03003ooooo00;oo`00=?oo00<00?ooool0[Ooo1P0O<Ooo00<00?ooool00_oo000dool0
0`00ooooo`2^ool501laool00`00ooooo`02ool003Coo`03003ooooo0;7oo`0301oooooo00Koo`80
7bSoo`03003ooooo00;oo`00=?oo00<00?ooool0^Ooo100O9ooo00<00?ooool00_oo000=ool40002
ool01@00oooooooo00001_oo10000ooo0P003Ooo00<00?ooool0/?oo0P0O1ooo0`0O5_oo0`0O3ooo
00<00?ooool00_oo000=ool00`00ooooo`04ool00`00ool00009ool01@00oooooooo00000_oo00<0
0?ooool02_oo00<00?ooool0/?oo0`0O1Ooo1@0O2Ooo0P0O2_oo0P0O4?oo00<00?ooool00_oo000>
ool00`00ooooo`04ool00`00ooooo`08ool01@00oooooooo00000_oo00<00?ooool02_oo00<00?oo
ool0[ooo1@0O1Ooo0P0O2Ooo1@0O2?oo1@0O3_oo00<00?ooool00_oo000?ool00`00ooooo`02ool0
0`00ool00009ool01@00oooooooo00000_oo00<00?ooool02_oo00<00?ooool0[ooo3@0O2Ooo1@0O
2Ooo0P0O3ooo00<00?ooool00_oo000@ool01000ooooo`000ooo00<00?ooool01_oo00D00?oooooo
o`0000;oo`03003ooooo00[oo`800;;oo``07aKoo`807`koo`8000Coo`003Ooo00@00?ooool000go
o`<000?oo`04003ooooo000<ool00`00ooooo`2_ool?01l8ool201l9ool501l>ool00`00ooooo`02
ool000koo`80013oo`03003ooooo00;oo`8000Soo`<000;oo`800;?oo``07`Ooo`D07`Soo`@07`ko
o`03003ooooo00;oo`00<?oo00L00?ooooooo`00ool0002jool501l7ool201l8ool701l=ool00`00
ooooo`02ool002[oo`<000?oo`07003oooooool00?oo0000^ooo100O1Ooo1@0O2Ooo1@0O3Ooo00<0
0?ooool00_oo000`ool01`00oooooooo003oo`000;_oo`D07`Ooo`807`Woo`H07`coo`03003ooooo
00;oo`00;ooo0P000ooo00<00?oo0000_Ooo00<07oooool01ooo0`0O2ooo2`0O1?oo00<00?ooool0
0_oo000`ool01000oooooooo0P00a_oo1@0O1ooo0`0O00?oo`0O01l0200O1Ooo00<00?ooool00_oo
000dool00`00ooooo`2mool201l7oolK01l3ool00`00ooooo`02ool003Coo`03003ooooo0;goo`03
01oooooo00Gooa007`?oo`H07`Koo`03003ooooo00;oo`00=?oo0P00_?oo1@0O1Ooo400O3_oo0P00
1?oo000dool00`00ooooo`2koolH01lAool00`00ooooo`02ool003Coo`03003ooooo0;coo``07aco
o`03003ooooo00;oo`00=?oo00<00?ooool0_?oo300O7?oo00<00?ooool00_oo000dool00`00oooo
o`3Tool00`00ooooo`02ool003Coo`03003ooooo0>Coo`03003ooooo00;oo`00=?oo00<00?ooool0
0_oo00<00?ooool0:Ooo00<00?ooool0:Ooo00<00?ooool0:Ooo00<00?ooool0:Ooo00<00?ooool0
:_oo00<00?ooool00_oo00<00?ooool00_oo000doooX0004ool00?ooob7oo`00oooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00
oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-26.3022, -3.79349*^-11, \
0.454326, 9.63028*^-13}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[12]="]
}, Open  ]],

Cell[TextData[{
  "The ",
  StyleBox["ImplicitSolver", "MR"],
  " method of ",
  StyleBox["ImplicitRungeKutta", "MR"],
  " has options ",
  StyleBox["AccuracyGoal", "MR"],
  " and ",
  StyleBox["PrecisionGoal", "MR"],
  " that specify the absolute and relative error to aim for in solving the \
nonlinear system of equations."
}], "Text"],

Cell[TextData[{
  "These options have the same default values as the corresponding options in \
",
  StyleBox["NDSolve", "MR"],
  ", since often there is little point in solving the nonlinear system to \
much higher accuracy than the local error of the method."
}], "Text"],

Cell["\<\
However, for certain types of problems it can be useful to solve \
the nonlinear system up to the working precision.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol\  = \ 
      NDSolve[system, \ 
        Method \[Rule] {"\<FixedStep\>", \ 
            Method \[Rule] {"\<ImplicitRungeKutta\>", "\<DifferenceOrder\>" \
\[Rule] 10, "\<ImplicitSolver\>" \[Rule] {"\<Newton\>", \ 
                    AccuracyGoal \[Rule] MachinePrecision, \ 
                    PrecisionGoal \[Rule] 
                      MachinePrecision, "\<IterationSafetyFactor\>" \[Rule] 
                      1}}}, \ StartingStepSize \[Rule] 1/10]\)], "Input",
  CellLabel->"In[13]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_3[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_4[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[13]="]
}, Open  ]],

Cell["\<\
The first invariant is the Hamiltonian of the system and the error \
is now bounded, as it should be since the Gauss implicit Runge-Kutta method \
is a symplectic integrator.\
\>", "Text"],

Cell["\<\
The second invariant is conserved exactly (up to roundoff) since \
the Gauss implicit Runge-Kutta method conserves quadratic invariants.\
\>", \
"Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InvariantErrorPlot[invs, \ vars, \ T, \ sol, \ 
      SymbolStyle \[Rule] {Red, \ Blue}, \ 
      InvariantErrorSampleRate \[Rule] 1]\)], "Input",
  CellLabel->"In[14]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.00952381 0.0238095 1.55038e+10 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.21429 -0.0125 -6 -9 ]
[.21429 -0.0125 6 0 ]
[.40476 -0.0125 -6 -9 ]
[.40476 -0.0125 6 0 ]
[.59524 -0.0125 -6 -9 ]
[.59524 -0.0125 6 0 ]
[.78571 -0.0125 -6 -9 ]
[.78571 -0.0125 6 0 ]
[.97619 -0.0125 -9 -9 ]
[.97619 -0.0125 9 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .02381 -6 -4.5 ]
[-0.0125 .02381 0 4.5 ]
[-0.0125 .17885 -37.5 -6.25 ]
[-0.0125 .17885 0 6.25 ]
[-0.0125 .33389 -37.5 -6.25 ]
[-0.0125 .33389 0 6.25 ]
[-0.0125 .48892 -37.5 -6.25 ]
[-0.0125 .48892 0 6.25 ]
[-0.0125 .64396 -37.5 -6.25 ]
[-0.0125 .64396 0 6.25 ]
[-0.0125 .799 -37.5 -6.25 ]
[-0.0125 .799 0 6.25 ]
[-0.0125 .95404 -37.5 -6.25 ]
[-0.0125 .95404 0 6.25 ]
[ 0 0 -0.125 0 ]
[ 0 1 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(20)] .21429 -0.0125 0 1 Mshowa
.40476 0 m
.40476 .00625 L
s
[(40)] .40476 -0.0125 0 1 Mshowa
.59524 0 m
.59524 .00625 L
s
[(60)] .59524 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(80)] .78571 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(100)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07143 0 m
.07143 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.16667 0 m
.16667 .00375 L
s
.2619 0 m
.2619 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.35714 0 m
.35714 .00375 L
s
.45238 0 m
.45238 .00375 L
s
.5 0 m
.5 .00375 L
s
.54762 0 m
.54762 .00375 L
s
.64286 0 m
.64286 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.7381 0 m
.7381 .00375 L
s
.83333 0 m
.83333 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .02381 m
.00625 .02381 L
s
[(0)] -0.0125 .02381 1 0 Mshowa
0 .17885 m
.00625 .17885 L
s
gsave
-0.0125 .17885 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .33389 m
.00625 .33389 L
s
gsave
-0.0125 .33389 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .48892 m
.00625 .48892 L
s
gsave
-0.0125 .48892 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(3) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .64396 m
.00625 .64396 L
s
gsave
-0.0125 .64396 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .799 m
.00625 .799 L
s
gsave
-0.0125 .799 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .95404 m
.00625 .95404 L
s
gsave
-0.0125 .95404 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(6) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
.125 Mabswid
0 .05482 m
.00375 .05482 L
s
0 .08582 m
.00375 .08582 L
s
0 .11683 m
.00375 .11683 L
s
0 .14784 m
.00375 .14784 L
s
0 .20986 m
.00375 .20986 L
s
0 .24086 m
.00375 .24086 L
s
0 .27187 m
.00375 .27187 L
s
0 .30288 m
.00375 .30288 L
s
0 .36489 m
.00375 .36489 L
s
0 .3959 m
.00375 .3959 L
s
0 .42691 m
.00375 .42691 L
s
0 .45792 m
.00375 .45792 L
s
0 .51993 m
.00375 .51993 L
s
0 .55094 m
.00375 .55094 L
s
0 .58195 m
.00375 .58195 L
s
0 .61295 m
.00375 .61295 L
s
0 .67497 m
.00375 .67497 L
s
0 .70598 m
.00375 .70598 L
s
0 .73698 m
.00375 .73698 L
s
0 .76799 m
.00375 .76799 L
s
0 .83001 m
.00375 .83001 L
s
0 .86101 m
.00375 .86101 L
s
0 .89202 m
.00375 .89202 L
s
0 .92303 m
.00375 .92303 L
s
0 .98505 m
.00375 .98505 L
s
.25 Mabswid
0 0 m
0 1 L
s
.02381 .99375 m
.02381 1 L
s
.21429 .99375 m
.21429 1 L
s
.40476 .99375 m
.40476 1 L
s
.59524 .99375 m
.59524 1 L
s
.78571 .99375 m
.78571 1 L
s
.97619 .99375 m
.97619 1 L
s
.125 Mabswid
.07143 .99625 m
.07143 1 L
s
.11905 .99625 m
.11905 1 L
s
.16667 .99625 m
.16667 1 L
s
.2619 .99625 m
.2619 1 L
s
.30952 .99625 m
.30952 1 L
s
.35714 .99625 m
.35714 1 L
s
.45238 .99625 m
.45238 1 L
s
.5 .99625 m
.5 1 L
s
.54762 .99625 m
.54762 1 L
s
.64286 .99625 m
.64286 1 L
s
.69048 .99625 m
.69048 1 L
s
.7381 .99625 m
.7381 1 L
s
.83333 .99625 m
.83333 1 L
s
.88095 .99625 m
.88095 1 L
s
.92857 .99625 m
.92857 1 L
s
.25 Mabswid
0 1 m
1 1 L
s
.99375 .02381 m
1 .02381 L
s
.99375 .17885 m
1 .17885 L
s
.99375 .33389 m
1 .33389 L
s
.99375 .48892 m
1 .48892 L
s
.99375 .64396 m
1 .64396 L
s
.99375 .799 m
1 .799 L
s
.99375 .95404 m
1 .95404 L
s
.125 Mabswid
.99625 .05482 m
1 .05482 L
s
.99625 .08582 m
1 .08582 L
s
.99625 .11683 m
1 .11683 L
s
.99625 .14784 m
1 .14784 L
s
.99625 .20986 m
1 .20986 L
s
.99625 .24086 m
1 .24086 L
s
.99625 .27187 m
1 .27187 L
s
.99625 .30288 m
1 .30288 L
s
.99625 .36489 m
1 .36489 L
s
.99625 .3959 m
1 .3959 L
s
.99625 .42691 m
1 .42691 L
s
.99625 .45792 m
1 .45792 L
s
.99625 .51993 m
1 .51993 L
s
.99625 .55094 m
1 .55094 L
s
.99625 .58195 m
1 .58195 L
s
.99625 .61295 m
1 .61295 L
s
.99625 .67497 m
1 .67497 L
s
.99625 .70598 m
1 .70598 L
s
.99625 .73698 m
1 .73698 L
s
.99625 .76799 m
1 .76799 L
s
.99625 .83001 m
1 .83001 L
s
.99625 .86101 m
1 .86101 L
s
.99625 .89202 m
1 .89202 L
s
.99625 .92303 m
1 .92303 L
s
.99625 .98505 m
1 .98505 L
s
.25 Mabswid
1 0 m
1 1 L
s
1 0 0 r
.0001 w
.02381 .02381 -1.875 0 Mabsadd m
.02381 .02381 0 2.5 Mabsadd L
.02381 .02381 1.875 0 Mabsadd L
.02381 .02381 0 -2.5 Mabsadd L
.02381 .02381 -1.875 0 Mabsadd L closepath F
.02476 .67126 -1.875 0 Mabsadd m
.02476 .67126 0 2.5 Mabsadd L
.02476 .67126 1.875 0 Mabsadd L
.02476 .67126 0 -2.5 Mabsadd L
.02476 .67126 -1.875 0 Mabsadd L closepath F
.02571 .64527 -1.875 0 Mabsadd m
.02571 .64527 0 2.5 Mabsadd L
.02571 .64527 1.875 0 Mabsadd L
.02571 .64527 0 -2.5 Mabsadd L
.02571 .64527 -1.875 0 Mabsadd L closepath F
.02667 .71615 -1.875 0 Mabsadd m
.02667 .71615 0 2.5 Mabsadd L
.02667 .71615 1.875 0 Mabsadd L
.02667 .71615 0 -2.5 Mabsadd L
.02667 .71615 -1.875 0 Mabsadd L closepath F
.02762 .72123 -1.875 0 Mabsadd m
.02762 .72123 0 2.5 Mabsadd L
.02762 .72123 1.875 0 Mabsadd L
.02762 .72123 0 -2.5 Mabsadd L
.02762 .72123 -1.875 0 Mabsadd L closepath F
.02857 .72144 -1.875 0 Mabsadd m
.02857 .72144 0 2.5 Mabsadd L
.02857 .72144 1.875 0 Mabsadd L
.02857 .72144 0 -2.5 Mabsadd L
.02857 .72144 -1.875 0 Mabsadd L closepath F
.02952 .72157 -1.875 0 Mabsadd m
.02952 .72157 0 2.5 Mabsadd L
.02952 .72157 1.875 0 Mabsadd L
.02952 .72157 0 -2.5 Mabsadd L
.02952 .72157 -1.875 0 Mabsadd L closepath F
.03048 .72163 -1.875 0 Mabsadd m
.03048 .72163 0 2.5 Mabsadd L
.03048 .72163 1.875 0 Mabsadd L
.03048 .72163 0 -2.5 Mabsadd L
.03048 .72163 -1.875 0 Mabsadd L closepath F
.03143 .72165 -1.875 0 Mabsadd m
.03143 .72165 0 2.5 Mabsadd L
.03143 .72165 1.875 0 Mabsadd L
.03143 .72165 0 -2.5 Mabsadd L
.03143 .72165 -1.875 0 Mabsadd L closepath F
.03238 .72166 -1.875 0 Mabsadd m
.03238 .72166 0 2.5 Mabsadd L
.03238 .72166 1.875 0 Mabsadd L
.03238 .72166 0 -2.5 Mabsadd L
.03238 .72166 -1.875 0 Mabsadd L closepath F
.03333 .72166 -1.875 0 Mabsadd m
.03333 .72166 0 2.5 Mabsadd L
.03333 .72166 1.875 0 Mabsadd L
.03333 .72166 0 -2.5 Mabsadd L
.03333 .72166 -1.875 0 Mabsadd L closepath F
.03429 .72166 -1.875 0 Mabsadd m
.03429 .72166 0 2.5 Mabsadd L
.03429 .72166 1.875 0 Mabsadd L
.03429 .72166 0 -2.5 Mabsadd L
.03429 .72166 -1.875 0 Mabsadd L closepath F
.03524 .72166 -1.875 0 Mabsadd m
.03524 .72166 0 2.5 Mabsadd L
.03524 .72166 1.875 0 Mabsadd L
.03524 .72166 0 -2.5 Mabsadd L
.03524 .72166 -1.875 0 Mabsadd L closepath F
.03619 .72166 -1.875 0 Mabsadd m
.03619 .72166 0 2.5 Mabsadd L
.03619 .72166 1.875 0 Mabsadd L
.03619 .72166 0 -2.5 Mabsadd L
.03619 .72166 -1.875 0 Mabsadd L closepath F
.03714 .72166 -1.875 0 Mabsadd m
.03714 .72166 0 2.5 Mabsadd L
.03714 .72166 1.875 0 Mabsadd L
.03714 .72166 0 -2.5 Mabsadd L
.03714 .72166 -1.875 0 Mabsadd L closepath F
.0381 .72166 -1.875 0 Mabsadd m
.0381 .72166 0 2.5 Mabsadd L
.0381 .72166 1.875 0 Mabsadd L
.0381 .72166 0 -2.5 Mabsadd L
.0381 .72166 -1.875 0 Mabsadd L closepath F
.03905 .72166 -1.875 0 Mabsadd m
.03905 .72166 0 2.5 Mabsadd L
.03905 .72166 1.875 0 Mabsadd L
.03905 .72166 0 -2.5 Mabsadd L
.03905 .72166 -1.875 0 Mabsadd L closepath F
.04 .72166 -1.875 0 Mabsadd m
.04 .72166 0 2.5 Mabsadd L
.04 .72166 1.875 0 Mabsadd L
.04 .72166 0 -2.5 Mabsadd L
.04 .72166 -1.875 0 Mabsadd L closepath F
.04095 .72166 -1.875 0 Mabsadd m
.04095 .72166 0 2.5 Mabsadd L
.04095 .72166 1.875 0 Mabsadd L
.04095 .72166 0 -2.5 Mabsadd L
.04095 .72166 -1.875 0 Mabsadd L closepath F
.0419 .72166 -1.875 0 Mabsadd m
.0419 .72166 0 2.5 Mabsadd L
.0419 .72166 1.875 0 Mabsadd L
.0419 .72166 0 -2.5 Mabsadd L
.0419 .72166 -1.875 0 Mabsadd L closepath F
.04286 .72166 -1.875 0 Mabsadd m
.04286 .72166 0 2.5 Mabsadd L
.04286 .72166 1.875 0 Mabsadd L
.04286 .72166 0 -2.5 Mabsadd L
.04286 .72166 -1.875 0 Mabsadd L closepath F
.04381 .72166 -1.875 0 Mabsadd m
.04381 .72166 0 2.5 Mabsadd L
.04381 .72166 1.875 0 Mabsadd L
.04381 .72166 0 -2.5 Mabsadd L
.04381 .72166 -1.875 0 Mabsadd L closepath F
.04476 .72166 -1.875 0 Mabsadd m
.04476 .72166 0 2.5 Mabsadd L
.04476 .72166 1.875 0 Mabsadd L
.04476 .72166 0 -2.5 Mabsadd L
.04476 .72166 -1.875 0 Mabsadd L closepath F
.04571 .72166 -1.875 0 Mabsadd m
.04571 .72166 0 2.5 Mabsadd L
.04571 .72166 1.875 0 Mabsadd L
.04571 .72166 0 -2.5 Mabsadd L
.04571 .72166 -1.875 0 Mabsadd L closepath F
.04667 .72166 -1.875 0 Mabsadd m
.04667 .72166 0 2.5 Mabsadd L
.04667 .72166 1.875 0 Mabsadd L
.04667 .72166 0 -2.5 Mabsadd L
.04667 .72166 -1.875 0 Mabsadd L closepath F
.04762 .72166 -1.875 0 Mabsadd m
.04762 .72166 0 2.5 Mabsadd L
.04762 .72166 1.875 0 Mabsadd L
.04762 .72166 0 -2.5 Mabsadd L
.04762 .72166 -1.875 0 Mabsadd L closepath F
.04857 .72166 -1.875 0 Mabsadd m
.04857 .72166 0 2.5 Mabsadd L
.04857 .72166 1.875 0 Mabsadd L
.04857 .72166 0 -2.5 Mabsadd L
.04857 .72166 -1.875 0 Mabsadd L closepath F
.04952 .72166 -1.875 0 Mabsadd m
.04952 .72166 0 2.5 Mabsadd L
.04952 .72166 1.875 0 Mabsadd L
.04952 .72166 0 -2.5 Mabsadd L
.04952 .72166 -1.875 0 Mabsadd L closepath F
.05048 .72166 -1.875 0 Mabsadd m
.05048 .72166 0 2.5 Mabsadd L
.05048 .72166 1.875 0 Mabsadd L
.05048 .72166 0 -2.5 Mabsadd L
.05048 .72166 -1.875 0 Mabsadd L closepath F
.05143 .72166 -1.875 0 Mabsadd m
.05143 .72166 0 2.5 Mabsadd L
.05143 .72166 1.875 0 Mabsadd L
.05143 .72166 0 -2.5 Mabsadd L
.05143 .72166 -1.875 0 Mabsadd L closepath F
.05238 .72166 -1.875 0 Mabsadd m
.05238 .72166 0 2.5 Mabsadd L
.05238 .72166 1.875 0 Mabsadd L
.05238 .72166 0 -2.5 Mabsadd L
.05238 .72166 -1.875 0 Mabsadd L closepath F
.05333 .72166 -1.875 0 Mabsadd m
.05333 .72166 0 2.5 Mabsadd L
.05333 .72166 1.875 0 Mabsadd L
.05333 .72166 0 -2.5 Mabsadd L
.05333 .72166 -1.875 0 Mabsadd L closepath F
.05429 .72166 -1.875 0 Mabsadd m
.05429 .72166 0 2.5 Mabsadd L
.05429 .72166 1.875 0 Mabsadd L
.05429 .72166 0 -2.5 Mabsadd L
.05429 .72166 -1.875 0 Mabsadd L closepath F
.05524 .72166 -1.875 0 Mabsadd m
.05524 .72166 0 2.5 Mabsadd L
.05524 .72166 1.875 0 Mabsadd L
.05524 .72166 0 -2.5 Mabsadd L
.05524 .72166 -1.875 0 Mabsadd L closepath F
.05619 .72166 -1.875 0 Mabsadd m
.05619 .72166 0 2.5 Mabsadd L
.05619 .72166 1.875 0 Mabsadd L
.05619 .72166 0 -2.5 Mabsadd L
.05619 .72166 -1.875 0 Mabsadd L closepath F
.05714 .72166 -1.875 0 Mabsadd m
.05714 .72166 0 2.5 Mabsadd L
.05714 .72166 1.875 0 Mabsadd L
.05714 .72166 0 -2.5 Mabsadd L
.05714 .72166 -1.875 0 Mabsadd L closepath F
.0581 .72166 -1.875 0 Mabsadd m
.0581 .72166 0 2.5 Mabsadd L
.0581 .72166 1.875 0 Mabsadd L
.0581 .72166 0 -2.5 Mabsadd L
.0581 .72166 -1.875 0 Mabsadd L closepath F
.05905 .72166 -1.875 0 Mabsadd m
.05905 .72166 0 2.5 Mabsadd L
.05905 .72166 1.875 0 Mabsadd L
.05905 .72166 0 -2.5 Mabsadd L
.05905 .72166 -1.875 0 Mabsadd L closepath F
.06 .72166 -1.875 0 Mabsadd m
.06 .72166 0 2.5 Mabsadd L
.06 .72166 1.875 0 Mabsadd L
.06 .72166 0 -2.5 Mabsadd L
.06 .72166 -1.875 0 Mabsadd L closepath F
.06095 .72166 -1.875 0 Mabsadd m
.06095 .72166 0 2.5 Mabsadd L
.06095 .72166 1.875 0 Mabsadd L
.06095 .72166 0 -2.5 Mabsadd L
.06095 .72166 -1.875 0 Mabsadd L closepath F
.0619 .72166 -1.875 0 Mabsadd m
.0619 .72166 0 2.5 Mabsadd L
.0619 .72166 1.875 0 Mabsadd L
.0619 .72166 0 -2.5 Mabsadd L
.0619 .72166 -1.875 0 Mabsadd L closepath F
.06286 .72166 -1.875 0 Mabsadd m
.06286 .72166 0 2.5 Mabsadd L
.06286 .72166 1.875 0 Mabsadd L
.06286 .72166 0 -2.5 Mabsadd L
.06286 .72166 -1.875 0 Mabsadd L closepath F
.06381 .72166 -1.875 0 Mabsadd m
.06381 .72166 0 2.5 Mabsadd L
.06381 .72166 1.875 0 Mabsadd L
.06381 .72166 0 -2.5 Mabsadd L
.06381 .72166 -1.875 0 Mabsadd L closepath F
.06476 .72166 -1.875 0 Mabsadd m
.06476 .72166 0 2.5 Mabsadd L
.06476 .72166 1.875 0 Mabsadd L
.06476 .72166 0 -2.5 Mabsadd L
.06476 .72166 -1.875 0 Mabsadd L closepath F
.06571 .72166 -1.875 0 Mabsadd m
.06571 .72166 0 2.5 Mabsadd L
.06571 .72166 1.875 0 Mabsadd L
.06571 .72166 0 -2.5 Mabsadd L
.06571 .72166 -1.875 0 Mabsadd L closepath F
.06667 .72166 -1.875 0 Mabsadd m
.06667 .72166 0 2.5 Mabsadd L
.06667 .72166 1.875 0 Mabsadd L
.06667 .72166 0 -2.5 Mabsadd L
.06667 .72166 -1.875 0 Mabsadd L closepath F
.06762 .72166 -1.875 0 Mabsadd m
.06762 .72166 0 2.5 Mabsadd L
.06762 .72166 1.875 0 Mabsadd L
.06762 .72166 0 -2.5 Mabsadd L
.06762 .72166 -1.875 0 Mabsadd L closepath F
.06857 .72166 -1.875 0 Mabsadd m
.06857 .72166 0 2.5 Mabsadd L
.06857 .72166 1.875 0 Mabsadd L
.06857 .72166 0 -2.5 Mabsadd L
.06857 .72166 -1.875 0 Mabsadd L closepath F
.06952 .72166 -1.875 0 Mabsadd m
.06952 .72166 0 2.5 Mabsadd L
.06952 .72166 1.875 0 Mabsadd L
.06952 .72166 0 -2.5 Mabsadd L
.06952 .72166 -1.875 0 Mabsadd L closepath F
.07048 .72163 -1.875 0 Mabsadd m
.07048 .72163 0 2.5 Mabsadd L
.07048 .72163 1.875 0 Mabsadd L
.07048 .72163 0 -2.5 Mabsadd L
.07048 .72163 -1.875 0 Mabsadd L closepath F
.07143 .72158 -1.875 0 Mabsadd m
.07143 .72158 0 2.5 Mabsadd L
.07143 .72158 1.875 0 Mabsadd L
.07143 .72158 0 -2.5 Mabsadd L
.07143 .72158 -1.875 0 Mabsadd L closepath F
.07238 .72145 -1.875 0 Mabsadd m
.07238 .72145 0 2.5 Mabsadd L
.07238 .72145 1.875 0 Mabsadd L
.07238 .72145 0 -2.5 Mabsadd L
.07238 .72145 -1.875 0 Mabsadd L closepath F
.07333 .72127 -1.875 0 Mabsadd m
.07333 .72127 0 2.5 Mabsadd L
.07333 .72127 1.875 0 Mabsadd L
.07333 .72127 0 -2.5 Mabsadd L
.07333 .72127 -1.875 0 Mabsadd L closepath F
.07429 .71807 -1.875 0 Mabsadd m
.07429 .71807 0 2.5 Mabsadd L
.07429 .71807 1.875 0 Mabsadd L
.07429 .71807 0 -2.5 Mabsadd L
.07429 .71807 -1.875 0 Mabsadd L closepath F
.07524 .66171 -1.875 0 Mabsadd m
.07524 .66171 0 2.5 Mabsadd L
.07524 .66171 1.875 0 Mabsadd L
.07524 .66171 0 -2.5 Mabsadd L
.07524 .66171 -1.875 0 Mabsadd L closepath F
.07619 .64328 -1.875 0 Mabsadd m
.07619 .64328 0 2.5 Mabsadd L
.07619 .64328 1.875 0 Mabsadd L
.07619 .64328 0 -2.5 Mabsadd L
.07619 .64328 -1.875 0 Mabsadd L closepath F
.07714 .11224 -1.875 0 Mabsadd m
.07714 .11224 0 2.5 Mabsadd L
.07714 .11224 1.875 0 Mabsadd L
.07714 .11224 0 -2.5 Mabsadd L
.07714 .11224 -1.875 0 Mabsadd L closepath F
.0781 .76594 -1.875 0 Mabsadd m
.0781 .76594 0 2.5 Mabsadd L
.0781 .76594 1.875 0 Mabsadd L
.0781 .76594 0 -2.5 Mabsadd L
.0781 .76594 -1.875 0 Mabsadd L closepath F
.07905 .70464 -1.875 0 Mabsadd m
.07905 .70464 0 2.5 Mabsadd L
.07905 .70464 1.875 0 Mabsadd L
.07905 .70464 0 -2.5 Mabsadd L
.07905 .70464 -1.875 0 Mabsadd L closepath F
.08 .78971 -1.875 0 Mabsadd m
.08 .78971 0 2.5 Mabsadd L
.08 .78971 1.875 0 Mabsadd L
.08 .78971 0 -2.5 Mabsadd L
.08 .78971 -1.875 0 Mabsadd L closepath F
.08095 .79758 -1.875 0 Mabsadd m
.08095 .79758 0 2.5 Mabsadd L
.08095 .79758 1.875 0 Mabsadd L
.08095 .79758 0 -2.5 Mabsadd L
.08095 .79758 -1.875 0 Mabsadd L closepath F
.0819 .79782 -1.875 0 Mabsadd m
.0819 .79782 0 2.5 Mabsadd L
.0819 .79782 1.875 0 Mabsadd L
.0819 .79782 0 -2.5 Mabsadd L
.0819 .79782 -1.875 0 Mabsadd L closepath F
.08286 .79797 -1.875 0 Mabsadd m
.08286 .79797 0 2.5 Mabsadd L
.08286 .79797 1.875 0 Mabsadd L
.08286 .79797 0 -2.5 Mabsadd L
.08286 .79797 -1.875 0 Mabsadd L closepath F
.08381 .79804 -1.875 0 Mabsadd m
.08381 .79804 0 2.5 Mabsadd L
.08381 .79804 1.875 0 Mabsadd L
.08381 .79804 0 -2.5 Mabsadd L
.08381 .79804 -1.875 0 Mabsadd L closepath F
.08476 .79807 -1.875 0 Mabsadd m
.08476 .79807 0 2.5 Mabsadd L
.08476 .79807 1.875 0 Mabsadd L
.08476 .79807 0 -2.5 Mabsadd L
.08476 .79807 -1.875 0 Mabsadd L closepath F
.08571 .79808 -1.875 0 Mabsadd m
.08571 .79808 0 2.5 Mabsadd L
.08571 .79808 1.875 0 Mabsadd L
.08571 .79808 0 -2.5 Mabsadd L
.08571 .79808 -1.875 0 Mabsadd L closepath F
.08667 .79808 -1.875 0 Mabsadd m
.08667 .79808 0 2.5 Mabsadd L
.08667 .79808 1.875 0 Mabsadd L
.08667 .79808 0 -2.5 Mabsadd L
.08667 .79808 -1.875 0 Mabsadd L closepath F
.08762 .79808 -1.875 0 Mabsadd m
.08762 .79808 0 2.5 Mabsadd L
.08762 .79808 1.875 0 Mabsadd L
.08762 .79808 0 -2.5 Mabsadd L
.08762 .79808 -1.875 0 Mabsadd L closepath F
.08857 .79808 -1.875 0 Mabsadd m
.08857 .79808 0 2.5 Mabsadd L
.08857 .79808 1.875 0 Mabsadd L
.08857 .79808 0 -2.5 Mabsadd L
.08857 .79808 -1.875 0 Mabsadd L closepath F
.08952 .79808 -1.875 0 Mabsadd m
.08952 .79808 0 2.5 Mabsadd L
.08952 .79808 1.875 0 Mabsadd L
.08952 .79808 0 -2.5 Mabsadd L
.08952 .79808 -1.875 0 Mabsadd L closepath F
.09048 .79808 -1.875 0 Mabsadd m
.09048 .79808 0 2.5 Mabsadd L
.09048 .79808 1.875 0 Mabsadd L
.09048 .79808 0 -2.5 Mabsadd L
.09048 .79808 -1.875 0 Mabsadd L closepath F
.09143 .79808 -1.875 0 Mabsadd m
.09143 .79808 0 2.5 Mabsadd L
.09143 .79808 1.875 0 Mabsadd L
.09143 .79808 0 -2.5 Mabsadd L
.09143 .79808 -1.875 0 Mabsadd L closepath F
.09238 .79808 -1.875 0 Mabsadd m
.09238 .79808 0 2.5 Mabsadd L
.09238 .79808 1.875 0 Mabsadd L
.09238 .79808 0 -2.5 Mabsadd L
.09238 .79808 -1.875 0 Mabsadd L closepath F
.09333 .79808 -1.875 0 Mabsadd m
.09333 .79808 0 2.5 Mabsadd L
.09333 .79808 1.875 0 Mabsadd L
.09333 .79808 0 -2.5 Mabsadd L
.09333 .79808 -1.875 0 Mabsadd L closepath F
.09429 .79808 -1.875 0 Mabsadd m
.09429 .79808 0 2.5 Mabsadd L
.09429 .79808 1.875 0 Mabsadd L
.09429 .79808 0 -2.5 Mabsadd L
.09429 .79808 -1.875 0 Mabsadd L closepath F
.09524 .79808 -1.875 0 Mabsadd m
.09524 .79808 0 2.5 Mabsadd L
.09524 .79808 1.875 0 Mabsadd L
.09524 .79808 0 -2.5 Mabsadd L
.09524 .79808 -1.875 0 Mabsadd L closepath F
.09619 .79808 -1.875 0 Mabsadd m
.09619 .79808 0 2.5 Mabsadd L
.09619 .79808 1.875 0 Mabsadd L
.09619 .79808 0 -2.5 Mabsadd L
.09619 .79808 -1.875 0 Mabsadd L closepath F
.09714 .79808 -1.875 0 Mabsadd m
.09714 .79808 0 2.5 Mabsadd L
.09714 .79808 1.875 0 Mabsadd L
.09714 .79808 0 -2.5 Mabsadd L
.09714 .79808 -1.875 0 Mabsadd L closepath F
.0981 .79808 -1.875 0 Mabsadd m
.0981 .79808 0 2.5 Mabsadd L
.0981 .79808 1.875 0 Mabsadd L
.0981 .79808 0 -2.5 Mabsadd L
.0981 .79808 -1.875 0 Mabsadd L closepath F
.09905 .79808 -1.875 0 Mabsadd m
.09905 .79808 0 2.5 Mabsadd L
.09905 .79808 1.875 0 Mabsadd L
.09905 .79808 0 -2.5 Mabsadd L
.09905 .79808 -1.875 0 Mabsadd L closepath F
.1 .79808 -1.875 0 Mabsadd m
.1 .79808 0 2.5 Mabsadd L
.1 .79808 1.875 0 Mabsadd L
.1 .79808 0 -2.5 Mabsadd L
.1 .79808 -1.875 0 Mabsadd L closepath F
.10095 .79807 -1.875 0 Mabsadd m
.10095 .79807 0 2.5 Mabsadd L
.10095 .79807 1.875 0 Mabsadd L
.10095 .79807 0 -2.5 Mabsadd L
.10095 .79807 -1.875 0 Mabsadd L closepath F
.1019 .79807 -1.875 0 Mabsadd m
.1019 .79807 0 2.5 Mabsadd L
.1019 .79807 1.875 0 Mabsadd L
.1019 .79807 0 -2.5 Mabsadd L
.1019 .79807 -1.875 0 Mabsadd L closepath F
.10286 .79807 -1.875 0 Mabsadd m
.10286 .79807 0 2.5 Mabsadd L
.10286 .79807 1.875 0 Mabsadd L
.10286 .79807 0 -2.5 Mabsadd L
.10286 .79807 -1.875 0 Mabsadd L closepath F
.10381 .79807 -1.875 0 Mabsadd m
.10381 .79807 0 2.5 Mabsadd L
.10381 .79807 1.875 0 Mabsadd L
.10381 .79807 0 -2.5 Mabsadd L
.10381 .79807 -1.875 0 Mabsadd L closepath F
.10476 .79807 -1.875 0 Mabsadd m
.10476 .79807 0 2.5 Mabsadd L
.10476 .79807 1.875 0 Mabsadd L
.10476 .79807 0 -2.5 Mabsadd L
.10476 .79807 -1.875 0 Mabsadd L closepath F
.10571 .79807 -1.875 0 Mabsadd m
.10571 .79807 0 2.5 Mabsadd L
.10571 .79807 1.875 0 Mabsadd L
.10571 .79807 0 -2.5 Mabsadd L
.10571 .79807 -1.875 0 Mabsadd L closepath F
.10667 .79807 -1.875 0 Mabsadd m
.10667 .79807 0 2.5 Mabsadd L
.10667 .79807 1.875 0 Mabsadd L
.10667 .79807 0 -2.5 Mabsadd L
.10667 .79807 -1.875 0 Mabsadd L closepath F
.10762 .79808 -1.875 0 Mabsadd m
.10762 .79808 0 2.5 Mabsadd L
.10762 .79808 1.875 0 Mabsadd L
.10762 .79808 0 -2.5 Mabsadd L
.10762 .79808 -1.875 0 Mabsadd L closepath F
.10857 .79808 -1.875 0 Mabsadd m
.10857 .79808 0 2.5 Mabsadd L
.10857 .79808 1.875 0 Mabsadd L
.10857 .79808 0 -2.5 Mabsadd L
.10857 .79808 -1.875 0 Mabsadd L closepath F
.10952 .79808 -1.875 0 Mabsadd m
.10952 .79808 0 2.5 Mabsadd L
.10952 .79808 1.875 0 Mabsadd L
.10952 .79808 0 -2.5 Mabsadd L
.10952 .79808 -1.875 0 Mabsadd L closepath F
.11048 .79808 -1.875 0 Mabsadd m
.11048 .79808 0 2.5 Mabsadd L
.11048 .79808 1.875 0 Mabsadd L
.11048 .79808 0 -2.5 Mabsadd L
.11048 .79808 -1.875 0 Mabsadd L closepath F
.11143 .79808 -1.875 0 Mabsadd m
.11143 .79808 0 2.5 Mabsadd L
.11143 .79808 1.875 0 Mabsadd L
.11143 .79808 0 -2.5 Mabsadd L
.11143 .79808 -1.875 0 Mabsadd L closepath F
.11238 .79808 -1.875 0 Mabsadd m
.11238 .79808 0 2.5 Mabsadd L
.11238 .79808 1.875 0 Mabsadd L
.11238 .79808 0 -2.5 Mabsadd L
.11238 .79808 -1.875 0 Mabsadd L closepath F
.11333 .79808 -1.875 0 Mabsadd m
.11333 .79808 0 2.5 Mabsadd L
.11333 .79808 1.875 0 Mabsadd L
.11333 .79808 0 -2.5 Mabsadd L
.11333 .79808 -1.875 0 Mabsadd L closepath F
.11429 .79808 -1.875 0 Mabsadd m
.11429 .79808 0 2.5 Mabsadd L
.11429 .79808 1.875 0 Mabsadd L
.11429 .79808 0 -2.5 Mabsadd L
.11429 .79808 -1.875 0 Mabsadd L closepath F
.11524 .79808 -1.875 0 Mabsadd m
.11524 .79808 0 2.5 Mabsadd L
.11524 .79808 1.875 0 Mabsadd L
.11524 .79808 0 -2.5 Mabsadd L
.11524 .79808 -1.875 0 Mabsadd L closepath F
.11619 .79808 -1.875 0 Mabsadd m
.11619 .79808 0 2.5 Mabsadd L
.11619 .79808 1.875 0 Mabsadd L
.11619 .79808 0 -2.5 Mabsadd L
.11619 .79808 -1.875 0 Mabsadd L closepath F
.11714 .79808 -1.875 0 Mabsadd m
.11714 .79808 0 2.5 Mabsadd L
.11714 .79808 1.875 0 Mabsadd L
.11714 .79808 0 -2.5 Mabsadd L
.11714 .79808 -1.875 0 Mabsadd L closepath F
.1181 .79807 -1.875 0 Mabsadd m
.1181 .79807 0 2.5 Mabsadd L
.1181 .79807 1.875 0 Mabsadd L
.1181 .79807 0 -2.5 Mabsadd L
.1181 .79807 -1.875 0 Mabsadd L closepath F
.11905 .79808 -1.875 0 Mabsadd m
.11905 .79808 0 2.5 Mabsadd L
.11905 .79808 1.875 0 Mabsadd L
.11905 .79808 0 -2.5 Mabsadd L
.11905 .79808 -1.875 0 Mabsadd L closepath F
.12 .79808 -1.875 0 Mabsadd m
.12 .79808 0 2.5 Mabsadd L
.12 .79808 1.875 0 Mabsadd L
.12 .79808 0 -2.5 Mabsadd L
.12 .79808 -1.875 0 Mabsadd L closepath F
.12095 .79807 -1.875 0 Mabsadd m
.12095 .79807 0 2.5 Mabsadd L
.12095 .79807 1.875 0 Mabsadd L
.12095 .79807 0 -2.5 Mabsadd L
.12095 .79807 -1.875 0 Mabsadd L closepath F
.1219 .79808 -1.875 0 Mabsadd m
.1219 .79808 0 2.5 Mabsadd L
.1219 .79808 1.875 0 Mabsadd L
.1219 .79808 0 -2.5 Mabsadd L
.1219 .79808 -1.875 0 Mabsadd L closepath F
.12286 .79807 -1.875 0 Mabsadd m
.12286 .79807 0 2.5 Mabsadd L
.12286 .79807 1.875 0 Mabsadd L
.12286 .79807 0 -2.5 Mabsadd L
.12286 .79807 -1.875 0 Mabsadd L closepath F
.12381 .79805 -1.875 0 Mabsadd m
.12381 .79805 0 2.5 Mabsadd L
.12381 .79805 1.875 0 Mabsadd L
.12381 .79805 0 -2.5 Mabsadd L
.12381 .79805 -1.875 0 Mabsadd L closepath F
.12476 .79801 -1.875 0 Mabsadd m
.12476 .79801 0 2.5 Mabsadd L
.12476 .79801 1.875 0 Mabsadd L
.12476 .79801 0 -2.5 Mabsadd L
.12476 .79801 -1.875 0 Mabsadd L closepath F
.12571 .79789 -1.875 0 Mabsadd m
.12571 .79789 0 2.5 Mabsadd L
.12571 .79789 1.875 0 Mabsadd L
.12571 .79789 0 -2.5 Mabsadd L
.12571 .79789 -1.875 0 Mabsadd L closepath F
.12667 .79772 -1.875 0 Mabsadd m
.12667 .79772 0 2.5 Mabsadd L
.12667 .79772 1.875 0 Mabsadd L
.12667 .79772 0 -2.5 Mabsadd L
.12667 .79772 -1.875 0 Mabsadd L closepath F
.12762 .79574 -1.875 0 Mabsadd m
.12762 .79574 0 2.5 Mabsadd L
.12762 .79574 1.875 0 Mabsadd L
.12762 .79574 0 -2.5 Mabsadd L
.12762 .79574 -1.875 0 Mabsadd L closepath F
.12857 .7527 -1.875 0 Mabsadd m
.12857 .7527 0 2.5 Mabsadd L
.12857 .7527 1.875 0 Mabsadd L
.12857 .7527 0 -2.5 Mabsadd L
.12857 .7527 -1.875 0 Mabsadd L closepath F
.12952 .69452 -1.875 0 Mabsadd m
.12952 .69452 0 2.5 Mabsadd L
.12952 .69452 1.875 0 Mabsadd L
.12952 .69452 0 -2.5 Mabsadd L
.12952 .69452 -1.875 0 Mabsadd L closepath F
.13048 .33884 -1.875 0 Mabsadd m
.13048 .33884 0 2.5 Mabsadd L
.13048 .33884 1.875 0 Mabsadd L
.13048 .33884 0 -2.5 Mabsadd L
.13048 .33884 -1.875 0 Mabsadd L closepath F
.13143 .8628 -1.875 0 Mabsadd m
.13143 .8628 0 2.5 Mabsadd L
.13143 .8628 1.875 0 Mabsadd L
.13143 .8628 0 -2.5 Mabsadd L
.13143 .8628 -1.875 0 Mabsadd L closepath F
.13238 .79412 -1.875 0 Mabsadd m
.13238 .79412 0 2.5 Mabsadd L
.13238 .79412 1.875 0 Mabsadd L
.13238 .79412 0 -2.5 Mabsadd L
.13238 .79412 -1.875 0 Mabsadd L closepath F
.13333 .89055 -1.875 0 Mabsadd m
.13333 .89055 0 2.5 Mabsadd L
.13333 .89055 1.875 0 Mabsadd L
.13333 .89055 0 -2.5 Mabsadd L
.13333 .89055 -1.875 0 Mabsadd L closepath F
.13429 .90244 -1.875 0 Mabsadd m
.13429 .90244 0 2.5 Mabsadd L
.13429 .90244 1.875 0 Mabsadd L
.13429 .90244 0 -2.5 Mabsadd L
.13429 .90244 -1.875 0 Mabsadd L closepath F
.13524 .90279 -1.875 0 Mabsadd m
.13524 .90279 0 2.5 Mabsadd L
.13524 .90279 1.875 0 Mabsadd L
.13524 .90279 0 -2.5 Mabsadd L
.13524 .90279 -1.875 0 Mabsadd L closepath F
.13619 .90294 -1.875 0 Mabsadd m
.13619 .90294 0 2.5 Mabsadd L
.13619 .90294 1.875 0 Mabsadd L
.13619 .90294 0 -2.5 Mabsadd L
.13619 .90294 -1.875 0 Mabsadd L closepath F
.13714 .90302 -1.875 0 Mabsadd m
.13714 .90302 0 2.5 Mabsadd L
.13714 .90302 1.875 0 Mabsadd L
.13714 .90302 0 -2.5 Mabsadd L
.13714 .90302 -1.875 0 Mabsadd L closepath F
.1381 .90305 -1.875 0 Mabsadd m
.1381 .90305 0 2.5 Mabsadd L
.1381 .90305 1.875 0 Mabsadd L
.1381 .90305 0 -2.5 Mabsadd L
.1381 .90305 -1.875 0 Mabsadd L closepath F
.13905 .90306 -1.875 0 Mabsadd m
.13905 .90306 0 2.5 Mabsadd L
.13905 .90306 1.875 0 Mabsadd L
.13905 .90306 0 -2.5 Mabsadd L
.13905 .90306 -1.875 0 Mabsadd L closepath F
.14 .90306 -1.875 0 Mabsadd m
.14 .90306 0 2.5 Mabsadd L
.14 .90306 1.875 0 Mabsadd L
.14 .90306 0 -2.5 Mabsadd L
.14 .90306 -1.875 0 Mabsadd L closepath F
.14095 .90306 -1.875 0 Mabsadd m
.14095 .90306 0 2.5 Mabsadd L
.14095 .90306 1.875 0 Mabsadd L
.14095 .90306 0 -2.5 Mabsadd L
.14095 .90306 -1.875 0 Mabsadd L closepath F
.1419 .90306 -1.875 0 Mabsadd m
.1419 .90306 0 2.5 Mabsadd L
.1419 .90306 1.875 0 Mabsadd L
.1419 .90306 0 -2.5 Mabsadd L
.1419 .90306 -1.875 0 Mabsadd L closepath F
.14286 .90306 -1.875 0 Mabsadd m
.14286 .90306 0 2.5 Mabsadd L
.14286 .90306 1.875 0 Mabsadd L
.14286 .90306 0 -2.5 Mabsadd L
.14286 .90306 -1.875 0 Mabsadd L closepath F
.14381 .90306 -1.875 0 Mabsadd m
.14381 .90306 0 2.5 Mabsadd L
.14381 .90306 1.875 0 Mabsadd L
.14381 .90306 0 -2.5 Mabsadd L
.14381 .90306 -1.875 0 Mabsadd L closepath F
.14476 .90306 -1.875 0 Mabsadd m
.14476 .90306 0 2.5 Mabsadd L
.14476 .90306 1.875 0 Mabsadd L
.14476 .90306 0 -2.5 Mabsadd L
.14476 .90306 -1.875 0 Mabsadd L closepath F
.14571 .90306 -1.875 0 Mabsadd m
.14571 .90306 0 2.5 Mabsadd L
.14571 .90306 1.875 0 Mabsadd L
.14571 .90306 0 -2.5 Mabsadd L
.14571 .90306 -1.875 0 Mabsadd L closepath F
.14667 .90306 -1.875 0 Mabsadd m
.14667 .90306 0 2.5 Mabsadd L
.14667 .90306 1.875 0 Mabsadd L
.14667 .90306 0 -2.5 Mabsadd L
.14667 .90306 -1.875 0 Mabsadd L closepath F
.14762 .90306 -1.875 0 Mabsadd m
.14762 .90306 0 2.5 Mabsadd L
.14762 .90306 1.875 0 Mabsadd L
.14762 .90306 0 -2.5 Mabsadd L
.14762 .90306 -1.875 0 Mabsadd L closepath F
.14857 .90306 -1.875 0 Mabsadd m
.14857 .90306 0 2.5 Mabsadd L
.14857 .90306 1.875 0 Mabsadd L
.14857 .90306 0 -2.5 Mabsadd L
.14857 .90306 -1.875 0 Mabsadd L closepath F
.14952 .90306 -1.875 0 Mabsadd m
.14952 .90306 0 2.5 Mabsadd L
.14952 .90306 1.875 0 Mabsadd L
.14952 .90306 0 -2.5 Mabsadd L
.14952 .90306 -1.875 0 Mabsadd L closepath F
.15048 .90306 -1.875 0 Mabsadd m
.15048 .90306 0 2.5 Mabsadd L
.15048 .90306 1.875 0 Mabsadd L
.15048 .90306 0 -2.5 Mabsadd L
.15048 .90306 -1.875 0 Mabsadd L closepath F
.15143 .90306 -1.875 0 Mabsadd m
.15143 .90306 0 2.5 Mabsadd L
.15143 .90306 1.875 0 Mabsadd L
.15143 .90306 0 -2.5 Mabsadd L
.15143 .90306 -1.875 0 Mabsadd L closepath F
.15238 .90306 -1.875 0 Mabsadd m
.15238 .90306 0 2.5 Mabsadd L
.15238 .90306 1.875 0 Mabsadd L
.15238 .90306 0 -2.5 Mabsadd L
.15238 .90306 -1.875 0 Mabsadd L closepath F
.15333 .90306 -1.875 0 Mabsadd m
.15333 .90306 0 2.5 Mabsadd L
.15333 .90306 1.875 0 Mabsadd L
.15333 .90306 0 -2.5 Mabsadd L
.15333 .90306 -1.875 0 Mabsadd L closepath F
.15429 .90306 -1.875 0 Mabsadd m
.15429 .90306 0 2.5 Mabsadd L
.15429 .90306 1.875 0 Mabsadd L
.15429 .90306 0 -2.5 Mabsadd L
.15429 .90306 -1.875 0 Mabsadd L closepath F
.15524 .90306 -1.875 0 Mabsadd m
.15524 .90306 0 2.5 Mabsadd L
.15524 .90306 1.875 0 Mabsadd L
.15524 .90306 0 -2.5 Mabsadd L
.15524 .90306 -1.875 0 Mabsadd L closepath F
.15619 .90306 -1.875 0 Mabsadd m
.15619 .90306 0 2.5 Mabsadd L
.15619 .90306 1.875 0 Mabsadd L
.15619 .90306 0 -2.5 Mabsadd L
.15619 .90306 -1.875 0 Mabsadd L closepath F
.15714 .90306 -1.875 0 Mabsadd m
.15714 .90306 0 2.5 Mabsadd L
.15714 .90306 1.875 0 Mabsadd L
.15714 .90306 0 -2.5 Mabsadd L
.15714 .90306 -1.875 0 Mabsadd L closepath F
.1581 .90306 -1.875 0 Mabsadd m
.1581 .90306 0 2.5 Mabsadd L
.1581 .90306 1.875 0 Mabsadd L
.1581 .90306 0 -2.5 Mabsadd L
.1581 .90306 -1.875 0 Mabsadd L closepath F
.15905 .90306 -1.875 0 Mabsadd m
.15905 .90306 0 2.5 Mabsadd L
.15905 .90306 1.875 0 Mabsadd L
.15905 .90306 0 -2.5 Mabsadd L
.15905 .90306 -1.875 0 Mabsadd L closepath F
.16 .90306 -1.875 0 Mabsadd m
.16 .90306 0 2.5 Mabsadd L
.16 .90306 1.875 0 Mabsadd L
.16 .90306 0 -2.5 Mabsadd L
.16 .90306 -1.875 0 Mabsadd L closepath F
.16095 .90306 -1.875 0 Mabsadd m
.16095 .90306 0 2.5 Mabsadd L
.16095 .90306 1.875 0 Mabsadd L
.16095 .90306 0 -2.5 Mabsadd L
.16095 .90306 -1.875 0 Mabsadd L closepath F
.1619 .90306 -1.875 0 Mabsadd m
.1619 .90306 0 2.5 Mabsadd L
.1619 .90306 1.875 0 Mabsadd L
.1619 .90306 0 -2.5 Mabsadd L
.1619 .90306 -1.875 0 Mabsadd L closepath F
.16286 .90306 -1.875 0 Mabsadd m
.16286 .90306 0 2.5 Mabsadd L
.16286 .90306 1.875 0 Mabsadd L
.16286 .90306 0 -2.5 Mabsadd L
.16286 .90306 -1.875 0 Mabsadd L closepath F
.16381 .90306 -1.875 0 Mabsadd m
.16381 .90306 0 2.5 Mabsadd L
.16381 .90306 1.875 0 Mabsadd L
.16381 .90306 0 -2.5 Mabsadd L
.16381 .90306 -1.875 0 Mabsadd L closepath F
.16476 .90306 -1.875 0 Mabsadd m
.16476 .90306 0 2.5 Mabsadd L
.16476 .90306 1.875 0 Mabsadd L
.16476 .90306 0 -2.5 Mabsadd L
.16476 .90306 -1.875 0 Mabsadd L closepath F
.16571 .90306 -1.875 0 Mabsadd m
.16571 .90306 0 2.5 Mabsadd L
.16571 .90306 1.875 0 Mabsadd L
.16571 .90306 0 -2.5 Mabsadd L
.16571 .90306 -1.875 0 Mabsadd L closepath F
.16667 .90306 -1.875 0 Mabsadd m
.16667 .90306 0 2.5 Mabsadd L
.16667 .90306 1.875 0 Mabsadd L
.16667 .90306 0 -2.5 Mabsadd L
.16667 .90306 -1.875 0 Mabsadd L closepath F
.16762 .90306 -1.875 0 Mabsadd m
.16762 .90306 0 2.5 Mabsadd L
.16762 .90306 1.875 0 Mabsadd L
.16762 .90306 0 -2.5 Mabsadd L
.16762 .90306 -1.875 0 Mabsadd L closepath F
.16857 .90307 -1.875 0 Mabsadd m
.16857 .90307 0 2.5 Mabsadd L
.16857 .90307 1.875 0 Mabsadd L
.16857 .90307 0 -2.5 Mabsadd L
.16857 .90307 -1.875 0 Mabsadd L closepath F
.16952 .90307 -1.875 0 Mabsadd m
.16952 .90307 0 2.5 Mabsadd L
.16952 .90307 1.875 0 Mabsadd L
.16952 .90307 0 -2.5 Mabsadd L
.16952 .90307 -1.875 0 Mabsadd L closepath F
.17048 .90306 -1.875 0 Mabsadd m
.17048 .90306 0 2.5 Mabsadd L
.17048 .90306 1.875 0 Mabsadd L
.17048 .90306 0 -2.5 Mabsadd L
.17048 .90306 -1.875 0 Mabsadd L closepath F
.17143 .90307 -1.875 0 Mabsadd m
.17143 .90307 0 2.5 Mabsadd L
.17143 .90307 1.875 0 Mabsadd L
.17143 .90307 0 -2.5 Mabsadd L
.17143 .90307 -1.875 0 Mabsadd L closepath F
.17238 .90307 -1.875 0 Mabsadd m
.17238 .90307 0 2.5 Mabsadd L
.17238 .90307 1.875 0 Mabsadd L
.17238 .90307 0 -2.5 Mabsadd L
.17238 .90307 -1.875 0 Mabsadd L closepath F
.17333 .90307 -1.875 0 Mabsadd m
.17333 .90307 0 2.5 Mabsadd L
.17333 .90307 1.875 0 Mabsadd L
.17333 .90307 0 -2.5 Mabsadd L
.17333 .90307 -1.875 0 Mabsadd L closepath F
.17429 .90307 -1.875 0 Mabsadd m
.17429 .90307 0 2.5 Mabsadd L
.17429 .90307 1.875 0 Mabsadd L
.17429 .90307 0 -2.5 Mabsadd L
.17429 .90307 -1.875 0 Mabsadd L closepath F
.17524 .90307 -1.875 0 Mabsadd m
.17524 .90307 0 2.5 Mabsadd L
.17524 .90307 1.875 0 Mabsadd L
.17524 .90307 0 -2.5 Mabsadd L
.17524 .90307 -1.875 0 Mabsadd L closepath F
.17619 .90306 -1.875 0 Mabsadd m
.17619 .90306 0 2.5 Mabsadd L
.17619 .90306 1.875 0 Mabsadd L
.17619 .90306 0 -2.5 Mabsadd L
.17619 .90306 -1.875 0 Mabsadd L closepath F
.17714 .90305 -1.875 0 Mabsadd m
.17714 .90305 0 2.5 Mabsadd L
.17714 .90305 1.875 0 Mabsadd L
.17714 .90305 0 -2.5 Mabsadd L
.17714 .90305 -1.875 0 Mabsadd L closepath F
.1781 .90301 -1.875 0 Mabsadd m
.1781 .90301 0 2.5 Mabsadd L
.1781 .90301 1.875 0 Mabsadd L
.1781 .90301 0 -2.5 Mabsadd L
.1781 .90301 -1.875 0 Mabsadd L closepath F
.17905 .9029 -1.875 0 Mabsadd m
.17905 .9029 0 2.5 Mabsadd L
.17905 .9029 1.875 0 Mabsadd L
.17905 .9029 0 -2.5 Mabsadd L
.17905 .9029 -1.875 0 Mabsadd L closepath F
.18 .90273 -1.875 0 Mabsadd m
.18 .90273 0 2.5 Mabsadd L
.18 .90273 1.875 0 Mabsadd L
.18 .90273 0 -2.5 Mabsadd L
.18 .90273 -1.875 0 Mabsadd L closepath F
.18095 .90152 -1.875 0 Mabsadd m
.18095 .90152 0 2.5 Mabsadd L
.18095 .90152 1.875 0 Mabsadd L
.18095 .90152 0 -2.5 Mabsadd L
.18095 .90152 -1.875 0 Mabsadd L closepath F
.1819 .86977 -1.875 0 Mabsadd m
.1819 .86977 0 2.5 Mabsadd L
.1819 .86977 1.875 0 Mabsadd L
.1819 .86977 0 -2.5 Mabsadd L
.1819 .86977 -1.875 0 Mabsadd L closepath F
.18286 .78392 -1.875 0 Mabsadd m
.18286 .78392 0 2.5 Mabsadd L
.18286 .78392 1.875 0 Mabsadd L
.18286 .78392 0 -2.5 Mabsadd L
.18286 .78392 -1.875 0 Mabsadd L closepath F
.18381 .60708 -1.875 0 Mabsadd m
.18381 .60708 0 2.5 Mabsadd L
.18381 .60708 1.875 0 Mabsadd L
.18381 .60708 0 -2.5 Mabsadd L
.18381 .60708 -1.875 0 Mabsadd L closepath F
.18476 .87977 -1.875 0 Mabsadd m
.18476 .87977 0 2.5 Mabsadd L
.18476 .87977 1.875 0 Mabsadd L
.18476 .87977 0 -2.5 Mabsadd L
.18476 .87977 -1.875 0 Mabsadd L closepath F
.18571 .85086 -1.875 0 Mabsadd m
.18571 .85086 0 2.5 Mabsadd L
.18571 .85086 1.875 0 Mabsadd L
.18571 .85086 0 -2.5 Mabsadd L
.18571 .85086 -1.875 0 Mabsadd L closepath F
.18667 .95255 -1.875 0 Mabsadd m
.18667 .95255 0 2.5 Mabsadd L
.18667 .95255 1.875 0 Mabsadd L
.18667 .95255 0 -2.5 Mabsadd L
.18667 .95255 -1.875 0 Mabsadd L closepath F
.18762 .97003 -1.875 0 Mabsadd m
.18762 .97003 0 2.5 Mabsadd L
.18762 .97003 1.875 0 Mabsadd L
.18762 .97003 0 -2.5 Mabsadd L
.18762 .97003 -1.875 0 Mabsadd L closepath F
.18857 .97056 -1.875 0 Mabsadd m
.18857 .97056 0 2.5 Mabsadd L
.18857 .97056 1.875 0 Mabsadd L
.18857 .97056 0 -2.5 Mabsadd L
.18857 .97056 -1.875 0 Mabsadd L closepath F
.18952 .97072 -1.875 0 Mabsadd m
.18952 .97072 0 2.5 Mabsadd L
.18952 .97072 1.875 0 Mabsadd L
.18952 .97072 0 -2.5 Mabsadd L
.18952 .97072 -1.875 0 Mabsadd L closepath F
.19048 .9708 -1.875 0 Mabsadd m
.19048 .9708 0 2.5 Mabsadd L
.19048 .9708 1.875 0 Mabsadd L
.19048 .9708 0 -2.5 Mabsadd L
.19048 .9708 -1.875 0 Mabsadd L closepath F
.19143 .97084 -1.875 0 Mabsadd m
.19143 .97084 0 2.5 Mabsadd L
.19143 .97084 1.875 0 Mabsadd L
.19143 .97084 0 -2.5 Mabsadd L
.19143 .97084 -1.875 0 Mabsadd L closepath F
.19238 .97085 -1.875 0 Mabsadd m
.19238 .97085 0 2.5 Mabsadd L
.19238 .97085 1.875 0 Mabsadd L
.19238 .97085 0 -2.5 Mabsadd L
.19238 .97085 -1.875 0 Mabsadd L closepath F
.19333 .97085 -1.875 0 Mabsadd m
.19333 .97085 0 2.5 Mabsadd L
.19333 .97085 1.875 0 Mabsadd L
.19333 .97085 0 -2.5 Mabsadd L
.19333 .97085 -1.875 0 Mabsadd L closepath F
.19429 .97085 -1.875 0 Mabsadd m
.19429 .97085 0 2.5 Mabsadd L
.19429 .97085 1.875 0 Mabsadd L
.19429 .97085 0 -2.5 Mabsadd L
.19429 .97085 -1.875 0 Mabsadd L closepath F
.19524 .97085 -1.875 0 Mabsadd m
.19524 .97085 0 2.5 Mabsadd L
.19524 .97085 1.875 0 Mabsadd L
.19524 .97085 0 -2.5 Mabsadd L
.19524 .97085 -1.875 0 Mabsadd L closepath F
.19619 .97085 -1.875 0 Mabsadd m
.19619 .97085 0 2.5 Mabsadd L
.19619 .97085 1.875 0 Mabsadd L
.19619 .97085 0 -2.5 Mabsadd L
.19619 .97085 -1.875 0 Mabsadd L closepath F
.19714 .97085 -1.875 0 Mabsadd m
.19714 .97085 0 2.5 Mabsadd L
.19714 .97085 1.875 0 Mabsadd L
.19714 .97085 0 -2.5 Mabsadd L
.19714 .97085 -1.875 0 Mabsadd L closepath F
.1981 .97085 -1.875 0 Mabsadd m
.1981 .97085 0 2.5 Mabsadd L
.1981 .97085 1.875 0 Mabsadd L
.1981 .97085 0 -2.5 Mabsadd L
.1981 .97085 -1.875 0 Mabsadd L closepath F
.19905 .97085 -1.875 0 Mabsadd m
.19905 .97085 0 2.5 Mabsadd L
.19905 .97085 1.875 0 Mabsadd L
.19905 .97085 0 -2.5 Mabsadd L
.19905 .97085 -1.875 0 Mabsadd L closepath F
.2 .97085 -1.875 0 Mabsadd m
.2 .97085 0 2.5 Mabsadd L
.2 .97085 1.875 0 Mabsadd L
.2 .97085 0 -2.5 Mabsadd L
.2 .97085 -1.875 0 Mabsadd L closepath F
.20095 .97085 -1.875 0 Mabsadd m
.20095 .97085 0 2.5 Mabsadd L
.20095 .97085 1.875 0 Mabsadd L
.20095 .97085 0 -2.5 Mabsadd L
.20095 .97085 -1.875 0 Mabsadd L closepath F
.2019 .97085 -1.875 0 Mabsadd m
.2019 .97085 0 2.5 Mabsadd L
.2019 .97085 1.875 0 Mabsadd L
.2019 .97085 0 -2.5 Mabsadd L
.2019 .97085 -1.875 0 Mabsadd L closepath F
.20286 .97085 -1.875 0 Mabsadd m
.20286 .97085 0 2.5 Mabsadd L
.20286 .97085 1.875 0 Mabsadd L
.20286 .97085 0 -2.5 Mabsadd L
.20286 .97085 -1.875 0 Mabsadd L closepath F
.20381 .97085 -1.875 0 Mabsadd m
.20381 .97085 0 2.5 Mabsadd L
.20381 .97085 1.875 0 Mabsadd L
.20381 .97085 0 -2.5 Mabsadd L
.20381 .97085 -1.875 0 Mabsadd L closepath F
.20476 .97085 -1.875 0 Mabsadd m
.20476 .97085 0 2.5 Mabsadd L
.20476 .97085 1.875 0 Mabsadd L
.20476 .97085 0 -2.5 Mabsadd L
.20476 .97085 -1.875 0 Mabsadd L closepath F
.20571 .97085 -1.875 0 Mabsadd m
.20571 .97085 0 2.5 Mabsadd L
.20571 .97085 1.875 0 Mabsadd L
.20571 .97085 0 -2.5 Mabsadd L
.20571 .97085 -1.875 0 Mabsadd L closepath F
.20667 .97085 -1.875 0 Mabsadd m
.20667 .97085 0 2.5 Mabsadd L
.20667 .97085 1.875 0 Mabsadd L
.20667 .97085 0 -2.5 Mabsadd L
.20667 .97085 -1.875 0 Mabsadd L closepath F
.20762 .97085 -1.875 0 Mabsadd m
.20762 .97085 0 2.5 Mabsadd L
.20762 .97085 1.875 0 Mabsadd L
.20762 .97085 0 -2.5 Mabsadd L
.20762 .97085 -1.875 0 Mabsadd L closepath F
.20857 .97085 -1.875 0 Mabsadd m
.20857 .97085 0 2.5 Mabsadd L
.20857 .97085 1.875 0 Mabsadd L
.20857 .97085 0 -2.5 Mabsadd L
.20857 .97085 -1.875 0 Mabsadd L closepath F
.20952 .97085 -1.875 0 Mabsadd m
.20952 .97085 0 2.5 Mabsadd L
.20952 .97085 1.875 0 Mabsadd L
.20952 .97085 0 -2.5 Mabsadd L
.20952 .97085 -1.875 0 Mabsadd L closepath F
.21048 .97085 -1.875 0 Mabsadd m
.21048 .97085 0 2.5 Mabsadd L
.21048 .97085 1.875 0 Mabsadd L
.21048 .97085 0 -2.5 Mabsadd L
.21048 .97085 -1.875 0 Mabsadd L closepath F
.21143 .97085 -1.875 0 Mabsadd m
.21143 .97085 0 2.5 Mabsadd L
.21143 .97085 1.875 0 Mabsadd L
.21143 .97085 0 -2.5 Mabsadd L
.21143 .97085 -1.875 0 Mabsadd L closepath F
.21238 .97085 -1.875 0 Mabsadd m
.21238 .97085 0 2.5 Mabsadd L
.21238 .97085 1.875 0 Mabsadd L
.21238 .97085 0 -2.5 Mabsadd L
.21238 .97085 -1.875 0 Mabsadd L closepath F
.21333 .97085 -1.875 0 Mabsadd m
.21333 .97085 0 2.5 Mabsadd L
.21333 .97085 1.875 0 Mabsadd L
.21333 .97085 0 -2.5 Mabsadd L
.21333 .97085 -1.875 0 Mabsadd L closepath F
.21429 .97085 -1.875 0 Mabsadd m
.21429 .97085 0 2.5 Mabsadd L
.21429 .97085 1.875 0 Mabsadd L
.21429 .97085 0 -2.5 Mabsadd L
.21429 .97085 -1.875 0 Mabsadd L closepath F
.21524 .97085 -1.875 0 Mabsadd m
.21524 .97085 0 2.5 Mabsadd L
.21524 .97085 1.875 0 Mabsadd L
.21524 .97085 0 -2.5 Mabsadd L
.21524 .97085 -1.875 0 Mabsadd L closepath F
.21619 .97085 -1.875 0 Mabsadd m
.21619 .97085 0 2.5 Mabsadd L
.21619 .97085 1.875 0 Mabsadd L
.21619 .97085 0 -2.5 Mabsadd L
.21619 .97085 -1.875 0 Mabsadd L closepath F
.21714 .97085 -1.875 0 Mabsadd m
.21714 .97085 0 2.5 Mabsadd L
.21714 .97085 1.875 0 Mabsadd L
.21714 .97085 0 -2.5 Mabsadd L
.21714 .97085 -1.875 0 Mabsadd L closepath F
.2181 .97086 -1.875 0 Mabsadd m
.2181 .97086 0 2.5 Mabsadd L
.2181 .97086 1.875 0 Mabsadd L
.2181 .97086 0 -2.5 Mabsadd L
.2181 .97086 -1.875 0 Mabsadd L closepath F
.21905 .97085 -1.875 0 Mabsadd m
.21905 .97085 0 2.5 Mabsadd L
.21905 .97085 1.875 0 Mabsadd L
.21905 .97085 0 -2.5 Mabsadd L
.21905 .97085 -1.875 0 Mabsadd L closepath F
.22 .97085 -1.875 0 Mabsadd m
.22 .97085 0 2.5 Mabsadd L
.22 .97085 1.875 0 Mabsadd L
.22 .97085 0 -2.5 Mabsadd L
.22 .97085 -1.875 0 Mabsadd L closepath F
.22095 .97085 -1.875 0 Mabsadd m
.22095 .97085 0 2.5 Mabsadd L
.22095 .97085 1.875 0 Mabsadd L
.22095 .97085 0 -2.5 Mabsadd L
.22095 .97085 -1.875 0 Mabsadd L closepath F
.2219 .97086 -1.875 0 Mabsadd m
.2219 .97086 0 2.5 Mabsadd L
.2219 .97086 1.875 0 Mabsadd L
.2219 .97086 0 -2.5 Mabsadd L
.2219 .97086 -1.875 0 Mabsadd L closepath F
.22286 .97086 -1.875 0 Mabsadd m
.22286 .97086 0 2.5 Mabsadd L
.22286 .97086 1.875 0 Mabsadd L
.22286 .97086 0 -2.5 Mabsadd L
.22286 .97086 -1.875 0 Mabsadd L closepath F
.22381 .97086 -1.875 0 Mabsadd m
.22381 .97086 0 2.5 Mabsadd L
.22381 .97086 1.875 0 Mabsadd L
.22381 .97086 0 -2.5 Mabsadd L
.22381 .97086 -1.875 0 Mabsadd L closepath F
.22476 .97086 -1.875 0 Mabsadd m
.22476 .97086 0 2.5 Mabsadd L
.22476 .97086 1.875 0 Mabsadd L
.22476 .97086 0 -2.5 Mabsadd L
.22476 .97086 -1.875 0 Mabsadd L closepath F
.22571 .97086 -1.875 0 Mabsadd m
.22571 .97086 0 2.5 Mabsadd L
.22571 .97086 1.875 0 Mabsadd L
.22571 .97086 0 -2.5 Mabsadd L
.22571 .97086 -1.875 0 Mabsadd L closepath F
.22667 .97086 -1.875 0 Mabsadd m
.22667 .97086 0 2.5 Mabsadd L
.22667 .97086 1.875 0 Mabsadd L
.22667 .97086 0 -2.5 Mabsadd L
.22667 .97086 -1.875 0 Mabsadd L closepath F
.22762 .97086 -1.875 0 Mabsadd m
.22762 .97086 0 2.5 Mabsadd L
.22762 .97086 1.875 0 Mabsadd L
.22762 .97086 0 -2.5 Mabsadd L
.22762 .97086 -1.875 0 Mabsadd L closepath F
.22857 .97085 -1.875 0 Mabsadd m
.22857 .97085 0 2.5 Mabsadd L
.22857 .97085 1.875 0 Mabsadd L
.22857 .97085 0 -2.5 Mabsadd L
.22857 .97085 -1.875 0 Mabsadd L closepath F
.22952 .97085 -1.875 0 Mabsadd m
.22952 .97085 0 2.5 Mabsadd L
.22952 .97085 1.875 0 Mabsadd L
.22952 .97085 0 -2.5 Mabsadd L
.22952 .97085 -1.875 0 Mabsadd L closepath F
.23048 .97084 -1.875 0 Mabsadd m
.23048 .97084 0 2.5 Mabsadd L
.23048 .97084 1.875 0 Mabsadd L
.23048 .97084 0 -2.5 Mabsadd L
.23048 .97084 -1.875 0 Mabsadd L closepath F
.23143 .97081 -1.875 0 Mabsadd m
.23143 .97081 0 2.5 Mabsadd L
.23143 .97081 1.875 0 Mabsadd L
.23143 .97081 0 -2.5 Mabsadd L
.23143 .97081 -1.875 0 Mabsadd L closepath F
.23238 .97071 -1.875 0 Mabsadd m
.23238 .97071 0 2.5 Mabsadd L
.23238 .97071 1.875 0 Mabsadd L
.23238 .97071 0 -2.5 Mabsadd L
.23238 .97071 -1.875 0 Mabsadd L closepath F
.23333 .97054 -1.875 0 Mabsadd m
.23333 .97054 0 2.5 Mabsadd L
.23333 .97054 1.875 0 Mabsadd L
.23333 .97054 0 -2.5 Mabsadd L
.23333 .97054 -1.875 0 Mabsadd L closepath F
.23429 .9698 -1.875 0 Mabsadd m
.23429 .9698 0 2.5 Mabsadd L
.23429 .9698 1.875 0 Mabsadd L
.23429 .9698 0 -2.5 Mabsadd L
.23429 .9698 -1.875 0 Mabsadd L closepath F
.23524 .94712 -1.875 0 Mabsadd m
.23524 .94712 0 2.5 Mabsadd L
.23524 .94712 1.875 0 Mabsadd L
.23524 .94712 0 -2.5 Mabsadd L
.23524 .94712 -1.875 0 Mabsadd L closepath F
.23619 .84754 -1.875 0 Mabsadd m
.23619 .84754 0 2.5 Mabsadd L
.23619 .84754 1.875 0 Mabsadd L
.23619 .84754 0 -2.5 Mabsadd L
.23619 .84754 -1.875 0 Mabsadd L closepath F
.23714 .81106 -1.875 0 Mabsadd m
.23714 .81106 0 2.5 Mabsadd L
.23714 .81106 1.875 0 Mabsadd L
.23714 .81106 0 -2.5 Mabsadd L
.23714 .81106 -1.875 0 Mabsadd L closepath F
.2381 .76567 -1.875 0 Mabsadd m
.2381 .76567 0 2.5 Mabsadd L
.2381 .76567 1.875 0 Mabsadd L
.2381 .76567 0 -2.5 Mabsadd L
.2381 .76567 -1.875 0 Mabsadd L closepath F
.23905 .83573 -1.875 0 Mabsadd m
.23905 .83573 0 2.5 Mabsadd L
.23905 .83573 1.875 0 Mabsadd L
.23905 .83573 0 -2.5 Mabsadd L
.23905 .83573 -1.875 0 Mabsadd L closepath F
.24 .93284 -1.875 0 Mabsadd m
.24 .93284 0 2.5 Mabsadd L
.24 .93284 1.875 0 Mabsadd L
.24 .93284 0 -2.5 Mabsadd L
.24 .93284 -1.875 0 Mabsadd L closepath F
.24095 .95781 -1.875 0 Mabsadd m
.24095 .95781 0 2.5 Mabsadd L
.24095 .95781 1.875 0 Mabsadd L
.24095 .95781 0 -2.5 Mabsadd L
.24095 .95781 -1.875 0 Mabsadd L closepath F
.2419 .95866 -1.875 0 Mabsadd m
.2419 .95866 0 2.5 Mabsadd L
.2419 .95866 1.875 0 Mabsadd L
.2419 .95866 0 -2.5 Mabsadd L
.2419 .95866 -1.875 0 Mabsadd L closepath F
.24286 .95883 -1.875 0 Mabsadd m
.24286 .95883 0 2.5 Mabsadd L
.24286 .95883 1.875 0 Mabsadd L
.24286 .95883 0 -2.5 Mabsadd L
.24286 .95883 -1.875 0 Mabsadd L closepath F
.24381 .95892 -1.875 0 Mabsadd m
.24381 .95892 0 2.5 Mabsadd L
.24381 .95892 1.875 0 Mabsadd L
.24381 .95892 0 -2.5 Mabsadd L
.24381 .95892 -1.875 0 Mabsadd L closepath F
.24476 .95896 -1.875 0 Mabsadd m
.24476 .95896 0 2.5 Mabsadd L
.24476 .95896 1.875 0 Mabsadd L
.24476 .95896 0 -2.5 Mabsadd L
.24476 .95896 -1.875 0 Mabsadd L closepath F
.24571 .95897 -1.875 0 Mabsadd m
.24571 .95897 0 2.5 Mabsadd L
.24571 .95897 1.875 0 Mabsadd L
.24571 .95897 0 -2.5 Mabsadd L
.24571 .95897 -1.875 0 Mabsadd L closepath F
.24667 .95898 -1.875 0 Mabsadd m
.24667 .95898 0 2.5 Mabsadd L
.24667 .95898 1.875 0 Mabsadd L
.24667 .95898 0 -2.5 Mabsadd L
.24667 .95898 -1.875 0 Mabsadd L closepath F
.24762 .95898 -1.875 0 Mabsadd m
.24762 .95898 0 2.5 Mabsadd L
.24762 .95898 1.875 0 Mabsadd L
.24762 .95898 0 -2.5 Mabsadd L
.24762 .95898 -1.875 0 Mabsadd L closepath F
.24857 .95897 -1.875 0 Mabsadd m
.24857 .95897 0 2.5 Mabsadd L
.24857 .95897 1.875 0 Mabsadd L
.24857 .95897 0 -2.5 Mabsadd L
.24857 .95897 -1.875 0 Mabsadd L closepath F
.24952 .95898 -1.875 0 Mabsadd m
.24952 .95898 0 2.5 Mabsadd L
.24952 .95898 1.875 0 Mabsadd L
.24952 .95898 0 -2.5 Mabsadd L
.24952 .95898 -1.875 0 Mabsadd L closepath F
.25048 .95897 -1.875 0 Mabsadd m
.25048 .95897 0 2.5 Mabsadd L
.25048 .95897 1.875 0 Mabsadd L
.25048 .95897 0 -2.5 Mabsadd L
.25048 .95897 -1.875 0 Mabsadd L closepath F
.25143 .95898 -1.875 0 Mabsadd m
.25143 .95898 0 2.5 Mabsadd L
.25143 .95898 1.875 0 Mabsadd L
.25143 .95898 0 -2.5 Mabsadd L
.25143 .95898 -1.875 0 Mabsadd L closepath F
.25238 .95898 -1.875 0 Mabsadd m
.25238 .95898 0 2.5 Mabsadd L
.25238 .95898 1.875 0 Mabsadd L
.25238 .95898 0 -2.5 Mabsadd L
.25238 .95898 -1.875 0 Mabsadd L closepath F
.25333 .95898 -1.875 0 Mabsadd m
.25333 .95898 0 2.5 Mabsadd L
.25333 .95898 1.875 0 Mabsadd L
.25333 .95898 0 -2.5 Mabsadd L
.25333 .95898 -1.875 0 Mabsadd L closepath F
.25429 .95898 -1.875 0 Mabsadd m
.25429 .95898 0 2.5 Mabsadd L
.25429 .95898 1.875 0 Mabsadd L
.25429 .95898 0 -2.5 Mabsadd L
.25429 .95898 -1.875 0 Mabsadd L closepath F
.25524 .95898 -1.875 0 Mabsadd m
.25524 .95898 0 2.5 Mabsadd L
.25524 .95898 1.875 0 Mabsadd L
.25524 .95898 0 -2.5 Mabsadd L
.25524 .95898 -1.875 0 Mabsadd L closepath F
.25619 .95898 -1.875 0 Mabsadd m
.25619 .95898 0 2.5 Mabsadd L
.25619 .95898 1.875 0 Mabsadd L
.25619 .95898 0 -2.5 Mabsadd L
.25619 .95898 -1.875 0 Mabsadd L closepath F
.25714 .95898 -1.875 0 Mabsadd m
.25714 .95898 0 2.5 Mabsadd L
.25714 .95898 1.875 0 Mabsadd L
.25714 .95898 0 -2.5 Mabsadd L
.25714 .95898 -1.875 0 Mabsadd L closepath F
.2581 .95898 -1.875 0 Mabsadd m
.2581 .95898 0 2.5 Mabsadd L
.2581 .95898 1.875 0 Mabsadd L
.2581 .95898 0 -2.5 Mabsadd L
.2581 .95898 -1.875 0 Mabsadd L closepath F
.25905 .95898 -1.875 0 Mabsadd m
.25905 .95898 0 2.5 Mabsadd L
.25905 .95898 1.875 0 Mabsadd L
.25905 .95898 0 -2.5 Mabsadd L
.25905 .95898 -1.875 0 Mabsadd L closepath F
.26 .95898 -1.875 0 Mabsadd m
.26 .95898 0 2.5 Mabsadd L
.26 .95898 1.875 0 Mabsadd L
.26 .95898 0 -2.5 Mabsadd L
.26 .95898 -1.875 0 Mabsadd L closepath F
.26095 .95898 -1.875 0 Mabsadd m
.26095 .95898 0 2.5 Mabsadd L
.26095 .95898 1.875 0 Mabsadd L
.26095 .95898 0 -2.5 Mabsadd L
.26095 .95898 -1.875 0 Mabsadd L closepath F
.2619 .95898 -1.875 0 Mabsadd m
.2619 .95898 0 2.5 Mabsadd L
.2619 .95898 1.875 0 Mabsadd L
.2619 .95898 0 -2.5 Mabsadd L
.2619 .95898 -1.875 0 Mabsadd L closepath F
.26286 .95898 -1.875 0 Mabsadd m
.26286 .95898 0 2.5 Mabsadd L
.26286 .95898 1.875 0 Mabsadd L
.26286 .95898 0 -2.5 Mabsadd L
.26286 .95898 -1.875 0 Mabsadd L closepath F
.26381 .95898 -1.875 0 Mabsadd m
.26381 .95898 0 2.5 Mabsadd L
.26381 .95898 1.875 0 Mabsadd L
.26381 .95898 0 -2.5 Mabsadd L
.26381 .95898 -1.875 0 Mabsadd L closepath F
.26476 .95898 -1.875 0 Mabsadd m
.26476 .95898 0 2.5 Mabsadd L
.26476 .95898 1.875 0 Mabsadd L
.26476 .95898 0 -2.5 Mabsadd L
.26476 .95898 -1.875 0 Mabsadd L closepath F
.26571 .95898 -1.875 0 Mabsadd m
.26571 .95898 0 2.5 Mabsadd L
.26571 .95898 1.875 0 Mabsadd L
.26571 .95898 0 -2.5 Mabsadd L
.26571 .95898 -1.875 0 Mabsadd L closepath F
.26667 .95898 -1.875 0 Mabsadd m
.26667 .95898 0 2.5 Mabsadd L
.26667 .95898 1.875 0 Mabsadd L
.26667 .95898 0 -2.5 Mabsadd L
.26667 .95898 -1.875 0 Mabsadd L closepath F
.26762 .95898 -1.875 0 Mabsadd m
.26762 .95898 0 2.5 Mabsadd L
.26762 .95898 1.875 0 Mabsadd L
.26762 .95898 0 -2.5 Mabsadd L
.26762 .95898 -1.875 0 Mabsadd L closepath F
.26857 .95898 -1.875 0 Mabsadd m
.26857 .95898 0 2.5 Mabsadd L
.26857 .95898 1.875 0 Mabsadd L
.26857 .95898 0 -2.5 Mabsadd L
.26857 .95898 -1.875 0 Mabsadd L closepath F
.26952 .95898 -1.875 0 Mabsadd m
.26952 .95898 0 2.5 Mabsadd L
.26952 .95898 1.875 0 Mabsadd L
.26952 .95898 0 -2.5 Mabsadd L
.26952 .95898 -1.875 0 Mabsadd L closepath F
.27048 .95898 -1.875 0 Mabsadd m
.27048 .95898 0 2.5 Mabsadd L
.27048 .95898 1.875 0 Mabsadd L
.27048 .95898 0 -2.5 Mabsadd L
.27048 .95898 -1.875 0 Mabsadd L closepath F
.27143 .95898 -1.875 0 Mabsadd m
.27143 .95898 0 2.5 Mabsadd L
.27143 .95898 1.875 0 Mabsadd L
.27143 .95898 0 -2.5 Mabsadd L
.27143 .95898 -1.875 0 Mabsadd L closepath F
.27238 .95898 -1.875 0 Mabsadd m
.27238 .95898 0 2.5 Mabsadd L
.27238 .95898 1.875 0 Mabsadd L
.27238 .95898 0 -2.5 Mabsadd L
.27238 .95898 -1.875 0 Mabsadd L closepath F
.27333 .95898 -1.875 0 Mabsadd m
.27333 .95898 0 2.5 Mabsadd L
.27333 .95898 1.875 0 Mabsadd L
.27333 .95898 0 -2.5 Mabsadd L
.27333 .95898 -1.875 0 Mabsadd L closepath F
.27429 .95898 -1.875 0 Mabsadd m
.27429 .95898 0 2.5 Mabsadd L
.27429 .95898 1.875 0 Mabsadd L
.27429 .95898 0 -2.5 Mabsadd L
.27429 .95898 -1.875 0 Mabsadd L closepath F
.27524 .95898 -1.875 0 Mabsadd m
.27524 .95898 0 2.5 Mabsadd L
.27524 .95898 1.875 0 Mabsadd L
.27524 .95898 0 -2.5 Mabsadd L
.27524 .95898 -1.875 0 Mabsadd L closepath F
.27619 .95898 -1.875 0 Mabsadd m
.27619 .95898 0 2.5 Mabsadd L
.27619 .95898 1.875 0 Mabsadd L
.27619 .95898 0 -2.5 Mabsadd L
.27619 .95898 -1.875 0 Mabsadd L closepath F
.27714 .95897 -1.875 0 Mabsadd m
.27714 .95897 0 2.5 Mabsadd L
.27714 .95897 1.875 0 Mabsadd L
.27714 .95897 0 -2.5 Mabsadd L
.27714 .95897 -1.875 0 Mabsadd L closepath F
.2781 .95898 -1.875 0 Mabsadd m
.2781 .95898 0 2.5 Mabsadd L
.2781 .95898 1.875 0 Mabsadd L
.2781 .95898 0 -2.5 Mabsadd L
.2781 .95898 -1.875 0 Mabsadd L closepath F
.27905 .95898 -1.875 0 Mabsadd m
.27905 .95898 0 2.5 Mabsadd L
.27905 .95898 1.875 0 Mabsadd L
.27905 .95898 0 -2.5 Mabsadd L
.27905 .95898 -1.875 0 Mabsadd L closepath F
.28 .95898 -1.875 0 Mabsadd m
.28 .95898 0 2.5 Mabsadd L
.28 .95898 1.875 0 Mabsadd L
.28 .95898 0 -2.5 Mabsadd L
.28 .95898 -1.875 0 Mabsadd L closepath F
.28095 .95898 -1.875 0 Mabsadd m
.28095 .95898 0 2.5 Mabsadd L
.28095 .95898 1.875 0 Mabsadd L
.28095 .95898 0 -2.5 Mabsadd L
.28095 .95898 -1.875 0 Mabsadd L closepath F
.2819 .95898 -1.875 0 Mabsadd m
.2819 .95898 0 2.5 Mabsadd L
.2819 .95898 1.875 0 Mabsadd L
.2819 .95898 0 -2.5 Mabsadd L
.2819 .95898 -1.875 0 Mabsadd L closepath F
.28286 .95897 -1.875 0 Mabsadd m
.28286 .95897 0 2.5 Mabsadd L
.28286 .95897 1.875 0 Mabsadd L
.28286 .95897 0 -2.5 Mabsadd L
.28286 .95897 -1.875 0 Mabsadd L closepath F
.28381 .95896 -1.875 0 Mabsadd m
.28381 .95896 0 2.5 Mabsadd L
.28381 .95896 1.875 0 Mabsadd L
.28381 .95896 0 -2.5 Mabsadd L
.28381 .95896 -1.875 0 Mabsadd L closepath F
.28476 .95893 -1.875 0 Mabsadd m
.28476 .95893 0 2.5 Mabsadd L
.28476 .95893 1.875 0 Mabsadd L
.28476 .95893 0 -2.5 Mabsadd L
.28476 .95893 -1.875 0 Mabsadd L closepath F
.28571 .95885 -1.875 0 Mabsadd m
.28571 .95885 0 2.5 Mabsadd L
.28571 .95885 1.875 0 Mabsadd L
.28571 .95885 0 -2.5 Mabsadd L
.28571 .95885 -1.875 0 Mabsadd L closepath F
.28667 .95869 -1.875 0 Mabsadd m
.28667 .95869 0 2.5 Mabsadd L
.28667 .95869 1.875 0 Mabsadd L
.28667 .95869 0 -2.5 Mabsadd L
.28667 .95869 -1.875 0 Mabsadd L closepath F
.28762 .95823 -1.875 0 Mabsadd m
.28762 .95823 0 2.5 Mabsadd L
.28762 .95823 1.875 0 Mabsadd L
.28762 .95823 0 -2.5 Mabsadd L
.28762 .95823 -1.875 0 Mabsadd L closepath F
.28857 .94248 -1.875 0 Mabsadd m
.28857 .94248 0 2.5 Mabsadd L
.28857 .94248 1.875 0 Mabsadd L
.28857 .94248 0 -2.5 Mabsadd L
.28857 .94248 -1.875 0 Mabsadd L closepath F
.28952 .84142 -1.875 0 Mabsadd m
.28952 .84142 0 2.5 Mabsadd L
.28952 .84142 1.875 0 Mabsadd L
.28952 .84142 0 -2.5 Mabsadd L
.28952 .84142 -1.875 0 Mabsadd L closepath F
.29048 .88703 -1.875 0 Mabsadd m
.29048 .88703 0 2.5 Mabsadd L
.29048 .88703 1.875 0 Mabsadd L
.29048 .88703 0 -2.5 Mabsadd L
.29048 .88703 -1.875 0 Mabsadd L closepath F
.29143 .53483 -1.875 0 Mabsadd m
.29143 .53483 0 2.5 Mabsadd L
.29143 .53483 1.875 0 Mabsadd L
.29143 .53483 0 -2.5 Mabsadd L
.29143 .53483 -1.875 0 Mabsadd L closepath F
.29238 .75894 -1.875 0 Mabsadd m
.29238 .75894 0 2.5 Mabsadd L
.29238 .75894 1.875 0 Mabsadd L
.29238 .75894 0 -2.5 Mabsadd L
.29238 .75894 -1.875 0 Mabsadd L closepath F
.29333 .83849 -1.875 0 Mabsadd m
.29333 .83849 0 2.5 Mabsadd L
.29333 .83849 1.875 0 Mabsadd L
.29333 .83849 0 -2.5 Mabsadd L
.29333 .83849 -1.875 0 Mabsadd L closepath F
.29429 .87316 -1.875 0 Mabsadd m
.29429 .87316 0 2.5 Mabsadd L
.29429 .87316 1.875 0 Mabsadd L
.29429 .87316 0 -2.5 Mabsadd L
.29429 .87316 -1.875 0 Mabsadd L closepath F
.29524 .87455 -1.875 0 Mabsadd m
.29524 .87455 0 2.5 Mabsadd L
.29524 .87455 1.875 0 Mabsadd L
.29524 .87455 0 -2.5 Mabsadd L
.29524 .87455 -1.875 0 Mabsadd L closepath F
.29619 .87472 -1.875 0 Mabsadd m
.29619 .87472 0 2.5 Mabsadd L
.29619 .87472 1.875 0 Mabsadd L
.29619 .87472 0 -2.5 Mabsadd L
.29619 .87472 -1.875 0 Mabsadd L closepath F
.29714 .87482 -1.875 0 Mabsadd m
.29714 .87482 0 2.5 Mabsadd L
.29714 .87482 1.875 0 Mabsadd L
.29714 .87482 0 -2.5 Mabsadd L
.29714 .87482 -1.875 0 Mabsadd L closepath F
.2981 .87486 -1.875 0 Mabsadd m
.2981 .87486 0 2.5 Mabsadd L
.2981 .87486 1.875 0 Mabsadd L
.2981 .87486 0 -2.5 Mabsadd L
.2981 .87486 -1.875 0 Mabsadd L closepath F
.29905 .87487 -1.875 0 Mabsadd m
.29905 .87487 0 2.5 Mabsadd L
.29905 .87487 1.875 0 Mabsadd L
.29905 .87487 0 -2.5 Mabsadd L
.29905 .87487 -1.875 0 Mabsadd L closepath F
.3 .87488 -1.875 0 Mabsadd m
.3 .87488 0 2.5 Mabsadd L
.3 .87488 1.875 0 Mabsadd L
.3 .87488 0 -2.5 Mabsadd L
.3 .87488 -1.875 0 Mabsadd L closepath F
.30095 .87488 -1.875 0 Mabsadd m
.30095 .87488 0 2.5 Mabsadd L
.30095 .87488 1.875 0 Mabsadd L
.30095 .87488 0 -2.5 Mabsadd L
.30095 .87488 -1.875 0 Mabsadd L closepath F
.3019 .87489 -1.875 0 Mabsadd m
.3019 .87489 0 2.5 Mabsadd L
.3019 .87489 1.875 0 Mabsadd L
.3019 .87489 0 -2.5 Mabsadd L
.3019 .87489 -1.875 0 Mabsadd L closepath F
.30286 .87488 -1.875 0 Mabsadd m
.30286 .87488 0 2.5 Mabsadd L
.30286 .87488 1.875 0 Mabsadd L
.30286 .87488 0 -2.5 Mabsadd L
.30286 .87488 -1.875 0 Mabsadd L closepath F
.30381 .87488 -1.875 0 Mabsadd m
.30381 .87488 0 2.5 Mabsadd L
.30381 .87488 1.875 0 Mabsadd L
.30381 .87488 0 -2.5 Mabsadd L
.30381 .87488 -1.875 0 Mabsadd L closepath F
.30476 .87488 -1.875 0 Mabsadd m
.30476 .87488 0 2.5 Mabsadd L
.30476 .87488 1.875 0 Mabsadd L
.30476 .87488 0 -2.5 Mabsadd L
.30476 .87488 -1.875 0 Mabsadd L closepath F
.30571 .87489 -1.875 0 Mabsadd m
.30571 .87489 0 2.5 Mabsadd L
.30571 .87489 1.875 0 Mabsadd L
.30571 .87489 0 -2.5 Mabsadd L
.30571 .87489 -1.875 0 Mabsadd L closepath F
.30667 .87488 -1.875 0 Mabsadd m
.30667 .87488 0 2.5 Mabsadd L
.30667 .87488 1.875 0 Mabsadd L
.30667 .87488 0 -2.5 Mabsadd L
.30667 .87488 -1.875 0 Mabsadd L closepath F
.30762 .87488 -1.875 0 Mabsadd m
.30762 .87488 0 2.5 Mabsadd L
.30762 .87488 1.875 0 Mabsadd L
.30762 .87488 0 -2.5 Mabsadd L
.30762 .87488 -1.875 0 Mabsadd L closepath F
.30857 .87489 -1.875 0 Mabsadd m
.30857 .87489 0 2.5 Mabsadd L
.30857 .87489 1.875 0 Mabsadd L
.30857 .87489 0 -2.5 Mabsadd L
.30857 .87489 -1.875 0 Mabsadd L closepath F
.30952 .87488 -1.875 0 Mabsadd m
.30952 .87488 0 2.5 Mabsadd L
.30952 .87488 1.875 0 Mabsadd L
.30952 .87488 0 -2.5 Mabsadd L
.30952 .87488 -1.875 0 Mabsadd L closepath F
.31048 .87488 -1.875 0 Mabsadd m
.31048 .87488 0 2.5 Mabsadd L
.31048 .87488 1.875 0 Mabsadd L
.31048 .87488 0 -2.5 Mabsadd L
.31048 .87488 -1.875 0 Mabsadd L closepath F
.31143 .87488 -1.875 0 Mabsadd m
.31143 .87488 0 2.5 Mabsadd L
.31143 .87488 1.875 0 Mabsadd L
.31143 .87488 0 -2.5 Mabsadd L
.31143 .87488 -1.875 0 Mabsadd L closepath F
.31238 .87488 -1.875 0 Mabsadd m
.31238 .87488 0 2.5 Mabsadd L
.31238 .87488 1.875 0 Mabsadd L
.31238 .87488 0 -2.5 Mabsadd L
.31238 .87488 -1.875 0 Mabsadd L closepath F
.31333 .87488 -1.875 0 Mabsadd m
.31333 .87488 0 2.5 Mabsadd L
.31333 .87488 1.875 0 Mabsadd L
.31333 .87488 0 -2.5 Mabsadd L
.31333 .87488 -1.875 0 Mabsadd L closepath F
.31429 .87488 -1.875 0 Mabsadd m
.31429 .87488 0 2.5 Mabsadd L
.31429 .87488 1.875 0 Mabsadd L
.31429 .87488 0 -2.5 Mabsadd L
.31429 .87488 -1.875 0 Mabsadd L closepath F
.31524 .87488 -1.875 0 Mabsadd m
.31524 .87488 0 2.5 Mabsadd L
.31524 .87488 1.875 0 Mabsadd L
.31524 .87488 0 -2.5 Mabsadd L
.31524 .87488 -1.875 0 Mabsadd L closepath F
.31619 .87488 -1.875 0 Mabsadd m
.31619 .87488 0 2.5 Mabsadd L
.31619 .87488 1.875 0 Mabsadd L
.31619 .87488 0 -2.5 Mabsadd L
.31619 .87488 -1.875 0 Mabsadd L closepath F
.31714 .87489 -1.875 0 Mabsadd m
.31714 .87489 0 2.5 Mabsadd L
.31714 .87489 1.875 0 Mabsadd L
.31714 .87489 0 -2.5 Mabsadd L
.31714 .87489 -1.875 0 Mabsadd L closepath F
.3181 .87488 -1.875 0 Mabsadd m
.3181 .87488 0 2.5 Mabsadd L
.3181 .87488 1.875 0 Mabsadd L
.3181 .87488 0 -2.5 Mabsadd L
.3181 .87488 -1.875 0 Mabsadd L closepath F
.31905 .87488 -1.875 0 Mabsadd m
.31905 .87488 0 2.5 Mabsadd L
.31905 .87488 1.875 0 Mabsadd L
.31905 .87488 0 -2.5 Mabsadd L
.31905 .87488 -1.875 0 Mabsadd L closepath F
.32 .87488 -1.875 0 Mabsadd m
.32 .87488 0 2.5 Mabsadd L
.32 .87488 1.875 0 Mabsadd L
.32 .87488 0 -2.5 Mabsadd L
.32 .87488 -1.875 0 Mabsadd L closepath F
.32095 .87488 -1.875 0 Mabsadd m
.32095 .87488 0 2.5 Mabsadd L
.32095 .87488 1.875 0 Mabsadd L
.32095 .87488 0 -2.5 Mabsadd L
.32095 .87488 -1.875 0 Mabsadd L closepath F
.3219 .87488 -1.875 0 Mabsadd m
.3219 .87488 0 2.5 Mabsadd L
.3219 .87488 1.875 0 Mabsadd L
.3219 .87488 0 -2.5 Mabsadd L
.3219 .87488 -1.875 0 Mabsadd L closepath F
.32286 .87488 -1.875 0 Mabsadd m
.32286 .87488 0 2.5 Mabsadd L
.32286 .87488 1.875 0 Mabsadd L
.32286 .87488 0 -2.5 Mabsadd L
.32286 .87488 -1.875 0 Mabsadd L closepath F
.32381 .87488 -1.875 0 Mabsadd m
.32381 .87488 0 2.5 Mabsadd L
.32381 .87488 1.875 0 Mabsadd L
.32381 .87488 0 -2.5 Mabsadd L
.32381 .87488 -1.875 0 Mabsadd L closepath F
.32476 .87488 -1.875 0 Mabsadd m
.32476 .87488 0 2.5 Mabsadd L
.32476 .87488 1.875 0 Mabsadd L
.32476 .87488 0 -2.5 Mabsadd L
.32476 .87488 -1.875 0 Mabsadd L closepath F
.32571 .87488 -1.875 0 Mabsadd m
.32571 .87488 0 2.5 Mabsadd L
.32571 .87488 1.875 0 Mabsadd L
.32571 .87488 0 -2.5 Mabsadd L
.32571 .87488 -1.875 0 Mabsadd L closepath F
.32667 .87488 -1.875 0 Mabsadd m
.32667 .87488 0 2.5 Mabsadd L
.32667 .87488 1.875 0 Mabsadd L
.32667 .87488 0 -2.5 Mabsadd L
.32667 .87488 -1.875 0 Mabsadd L closepath F
.32762 .87488 -1.875 0 Mabsadd m
.32762 .87488 0 2.5 Mabsadd L
.32762 .87488 1.875 0 Mabsadd L
.32762 .87488 0 -2.5 Mabsadd L
.32762 .87488 -1.875 0 Mabsadd L closepath F
.32857 .87488 -1.875 0 Mabsadd m
.32857 .87488 0 2.5 Mabsadd L
.32857 .87488 1.875 0 Mabsadd L
.32857 .87488 0 -2.5 Mabsadd L
.32857 .87488 -1.875 0 Mabsadd L closepath F
.32952 .87488 -1.875 0 Mabsadd m
.32952 .87488 0 2.5 Mabsadd L
.32952 .87488 1.875 0 Mabsadd L
.32952 .87488 0 -2.5 Mabsadd L
.32952 .87488 -1.875 0 Mabsadd L closepath F
.33048 .87489 -1.875 0 Mabsadd m
.33048 .87489 0 2.5 Mabsadd L
.33048 .87489 1.875 0 Mabsadd L
.33048 .87489 0 -2.5 Mabsadd L
.33048 .87489 -1.875 0 Mabsadd L closepath F
.33143 .87488 -1.875 0 Mabsadd m
.33143 .87488 0 2.5 Mabsadd L
.33143 .87488 1.875 0 Mabsadd L
.33143 .87488 0 -2.5 Mabsadd L
.33143 .87488 -1.875 0 Mabsadd L closepath F
.33238 .87488 -1.875 0 Mabsadd m
.33238 .87488 0 2.5 Mabsadd L
.33238 .87488 1.875 0 Mabsadd L
.33238 .87488 0 -2.5 Mabsadd L
.33238 .87488 -1.875 0 Mabsadd L closepath F
.33333 .87489 -1.875 0 Mabsadd m
.33333 .87489 0 2.5 Mabsadd L
.33333 .87489 1.875 0 Mabsadd L
.33333 .87489 0 -2.5 Mabsadd L
.33333 .87489 -1.875 0 Mabsadd L closepath F
.33429 .87488 -1.875 0 Mabsadd m
.33429 .87488 0 2.5 Mabsadd L
.33429 .87488 1.875 0 Mabsadd L
.33429 .87488 0 -2.5 Mabsadd L
.33429 .87488 -1.875 0 Mabsadd L closepath F
.33524 .87488 -1.875 0 Mabsadd m
.33524 .87488 0 2.5 Mabsadd L
.33524 .87488 1.875 0 Mabsadd L
.33524 .87488 0 -2.5 Mabsadd L
.33524 .87488 -1.875 0 Mabsadd L closepath F
.33619 .87488 -1.875 0 Mabsadd m
.33619 .87488 0 2.5 Mabsadd L
.33619 .87488 1.875 0 Mabsadd L
.33619 .87488 0 -2.5 Mabsadd L
.33619 .87488 -1.875 0 Mabsadd L closepath F
.33714 .87487 -1.875 0 Mabsadd m
.33714 .87487 0 2.5 Mabsadd L
.33714 .87487 1.875 0 Mabsadd L
.33714 .87487 0 -2.5 Mabsadd L
.33714 .87487 -1.875 0 Mabsadd L closepath F
.3381 .87485 -1.875 0 Mabsadd m
.3381 .87485 0 2.5 Mabsadd L
.3381 .87485 1.875 0 Mabsadd L
.3381 .87485 0 -2.5 Mabsadd L
.3381 .87485 -1.875 0 Mabsadd L closepath F
.33905 .87477 -1.875 0 Mabsadd m
.33905 .87477 0 2.5 Mabsadd L
.33905 .87477 1.875 0 Mabsadd L
.33905 .87477 0 -2.5 Mabsadd L
.33905 .87477 -1.875 0 Mabsadd L closepath F
.34 .87462 -1.875 0 Mabsadd m
.34 .87462 0 2.5 Mabsadd L
.34 .87462 1.875 0 Mabsadd L
.34 .87462 0 -2.5 Mabsadd L
.34 .87462 -1.875 0 Mabsadd L closepath F
.34095 .8743 -1.875 0 Mabsadd m
.34095 .8743 0 2.5 Mabsadd L
.34095 .8743 1.875 0 Mabsadd L
.34095 .8743 0 -2.5 Mabsadd L
.34095 .8743 -1.875 0 Mabsadd L closepath F
.3419 .86368 -1.875 0 Mabsadd m
.3419 .86368 0 2.5 Mabsadd L
.3419 .86368 1.875 0 Mabsadd L
.3419 .86368 0 -2.5 Mabsadd L
.3419 .86368 -1.875 0 Mabsadd L closepath F
.34286 .76993 -1.875 0 Mabsadd m
.34286 .76993 0 2.5 Mabsadd L
.34286 .76993 1.875 0 Mabsadd L
.34286 .76993 0 -2.5 Mabsadd L
.34286 .76993 -1.875 0 Mabsadd L closepath F
.34381 .84053 -1.875 0 Mabsadd m
.34381 .84053 0 2.5 Mabsadd L
.34381 .84053 1.875 0 Mabsadd L
.34381 .84053 0 -2.5 Mabsadd L
.34381 .84053 -1.875 0 Mabsadd L closepath F
.34476 .26689 -1.875 0 Mabsadd m
.34476 .26689 0 2.5 Mabsadd L
.34476 .26689 1.875 0 Mabsadd L
.34476 .26689 0 -2.5 Mabsadd L
.34476 .26689 -1.875 0 Mabsadd L closepath F
.34571 .6739 -1.875 0 Mabsadd m
.34571 .6739 0 2.5 Mabsadd L
.34571 .6739 1.875 0 Mabsadd L
.34571 .6739 0 -2.5 Mabsadd L
.34571 .6739 -1.875 0 Mabsadd L closepath F
.34667 .72204 -1.875 0 Mabsadd m
.34667 .72204 0 2.5 Mabsadd L
.34667 .72204 1.875 0 Mabsadd L
.34667 .72204 0 -2.5 Mabsadd L
.34667 .72204 -1.875 0 Mabsadd L closepath F
.34762 .7686 -1.875 0 Mabsadd m
.34762 .7686 0 2.5 Mabsadd L
.34762 .7686 1.875 0 Mabsadd L
.34762 .7686 0 -2.5 Mabsadd L
.34762 .7686 -1.875 0 Mabsadd L closepath F
.34857 .77087 -1.875 0 Mabsadd m
.34857 .77087 0 2.5 Mabsadd L
.34857 .77087 1.875 0 Mabsadd L
.34857 .77087 0 -2.5 Mabsadd L
.34857 .77087 -1.875 0 Mabsadd L closepath F
.34952 .77105 -1.875 0 Mabsadd m
.34952 .77105 0 2.5 Mabsadd L
.34952 .77105 1.875 0 Mabsadd L
.34952 .77105 0 -2.5 Mabsadd L
.34952 .77105 -1.875 0 Mabsadd L closepath F
.35048 .77116 -1.875 0 Mabsadd m
.35048 .77116 0 2.5 Mabsadd L
.35048 .77116 1.875 0 Mabsadd L
.35048 .77116 0 -2.5 Mabsadd L
.35048 .77116 -1.875 0 Mabsadd L closepath F
.35143 .77121 -1.875 0 Mabsadd m
.35143 .77121 0 2.5 Mabsadd L
.35143 .77121 1.875 0 Mabsadd L
.35143 .77121 0 -2.5 Mabsadd L
.35143 .77121 -1.875 0 Mabsadd L closepath F
.35238 .77122 -1.875 0 Mabsadd m
.35238 .77122 0 2.5 Mabsadd L
.35238 .77122 1.875 0 Mabsadd L
.35238 .77122 0 -2.5 Mabsadd L
.35238 .77122 -1.875 0 Mabsadd L closepath F
.35333 .77123 -1.875 0 Mabsadd m
.35333 .77123 0 2.5 Mabsadd L
.35333 .77123 1.875 0 Mabsadd L
.35333 .77123 0 -2.5 Mabsadd L
.35333 .77123 -1.875 0 Mabsadd L closepath F
.35429 .77123 -1.875 0 Mabsadd m
.35429 .77123 0 2.5 Mabsadd L
.35429 .77123 1.875 0 Mabsadd L
.35429 .77123 0 -2.5 Mabsadd L
.35429 .77123 -1.875 0 Mabsadd L closepath F
.35524 .77123 -1.875 0 Mabsadd m
.35524 .77123 0 2.5 Mabsadd L
.35524 .77123 1.875 0 Mabsadd L
.35524 .77123 0 -2.5 Mabsadd L
.35524 .77123 -1.875 0 Mabsadd L closepath F
.35619 .77123 -1.875 0 Mabsadd m
.35619 .77123 0 2.5 Mabsadd L
.35619 .77123 1.875 0 Mabsadd L
.35619 .77123 0 -2.5 Mabsadd L
.35619 .77123 -1.875 0 Mabsadd L closepath F
.35714 .77124 -1.875 0 Mabsadd m
.35714 .77124 0 2.5 Mabsadd L
.35714 .77124 1.875 0 Mabsadd L
.35714 .77124 0 -2.5 Mabsadd L
.35714 .77124 -1.875 0 Mabsadd L closepath F
.3581 .77123 -1.875 0 Mabsadd m
.3581 .77123 0 2.5 Mabsadd L
.3581 .77123 1.875 0 Mabsadd L
.3581 .77123 0 -2.5 Mabsadd L
.3581 .77123 -1.875 0 Mabsadd L closepath F
.35905 .77123 -1.875 0 Mabsadd m
.35905 .77123 0 2.5 Mabsadd L
.35905 .77123 1.875 0 Mabsadd L
.35905 .77123 0 -2.5 Mabsadd L
.35905 .77123 -1.875 0 Mabsadd L closepath F
.36 .77123 -1.875 0 Mabsadd m
.36 .77123 0 2.5 Mabsadd L
.36 .77123 1.875 0 Mabsadd L
.36 .77123 0 -2.5 Mabsadd L
.36 .77123 -1.875 0 Mabsadd L closepath F
.36095 .77123 -1.875 0 Mabsadd m
.36095 .77123 0 2.5 Mabsadd L
.36095 .77123 1.875 0 Mabsadd L
.36095 .77123 0 -2.5 Mabsadd L
.36095 .77123 -1.875 0 Mabsadd L closepath F
.3619 .77123 -1.875 0 Mabsadd m
.3619 .77123 0 2.5 Mabsadd L
.3619 .77123 1.875 0 Mabsadd L
.3619 .77123 0 -2.5 Mabsadd L
.3619 .77123 -1.875 0 Mabsadd L closepath F
.36286 .77123 -1.875 0 Mabsadd m
.36286 .77123 0 2.5 Mabsadd L
.36286 .77123 1.875 0 Mabsadd L
.36286 .77123 0 -2.5 Mabsadd L
.36286 .77123 -1.875 0 Mabsadd L closepath F
.36381 .77123 -1.875 0 Mabsadd m
.36381 .77123 0 2.5 Mabsadd L
.36381 .77123 1.875 0 Mabsadd L
.36381 .77123 0 -2.5 Mabsadd L
.36381 .77123 -1.875 0 Mabsadd L closepath F
.36476 .77123 -1.875 0 Mabsadd m
.36476 .77123 0 2.5 Mabsadd L
.36476 .77123 1.875 0 Mabsadd L
.36476 .77123 0 -2.5 Mabsadd L
.36476 .77123 -1.875 0 Mabsadd L closepath F
.36571 .77123 -1.875 0 Mabsadd m
.36571 .77123 0 2.5 Mabsadd L
.36571 .77123 1.875 0 Mabsadd L
.36571 .77123 0 -2.5 Mabsadd L
.36571 .77123 -1.875 0 Mabsadd L closepath F
.36667 .77123 -1.875 0 Mabsadd m
.36667 .77123 0 2.5 Mabsadd L
.36667 .77123 1.875 0 Mabsadd L
.36667 .77123 0 -2.5 Mabsadd L
.36667 .77123 -1.875 0 Mabsadd L closepath F
.36762 .77124 -1.875 0 Mabsadd m
.36762 .77124 0 2.5 Mabsadd L
.36762 .77124 1.875 0 Mabsadd L
.36762 .77124 0 -2.5 Mabsadd L
.36762 .77124 -1.875 0 Mabsadd L closepath F
.36857 .77123 -1.875 0 Mabsadd m
.36857 .77123 0 2.5 Mabsadd L
.36857 .77123 1.875 0 Mabsadd L
.36857 .77123 0 -2.5 Mabsadd L
.36857 .77123 -1.875 0 Mabsadd L closepath F
.36952 .77123 -1.875 0 Mabsadd m
.36952 .77123 0 2.5 Mabsadd L
.36952 .77123 1.875 0 Mabsadd L
.36952 .77123 0 -2.5 Mabsadd L
.36952 .77123 -1.875 0 Mabsadd L closepath F
.37048 .77123 -1.875 0 Mabsadd m
.37048 .77123 0 2.5 Mabsadd L
.37048 .77123 1.875 0 Mabsadd L
.37048 .77123 0 -2.5 Mabsadd L
.37048 .77123 -1.875 0 Mabsadd L closepath F
.37143 .77123 -1.875 0 Mabsadd m
.37143 .77123 0 2.5 Mabsadd L
.37143 .77123 1.875 0 Mabsadd L
.37143 .77123 0 -2.5 Mabsadd L
.37143 .77123 -1.875 0 Mabsadd L closepath F
.37238 .77123 -1.875 0 Mabsadd m
.37238 .77123 0 2.5 Mabsadd L
.37238 .77123 1.875 0 Mabsadd L
.37238 .77123 0 -2.5 Mabsadd L
.37238 .77123 -1.875 0 Mabsadd L closepath F
.37333 .77123 -1.875 0 Mabsadd m
.37333 .77123 0 2.5 Mabsadd L
.37333 .77123 1.875 0 Mabsadd L
.37333 .77123 0 -2.5 Mabsadd L
.37333 .77123 -1.875 0 Mabsadd L closepath F
.37429 .77123 -1.875 0 Mabsadd m
.37429 .77123 0 2.5 Mabsadd L
.37429 .77123 1.875 0 Mabsadd L
.37429 .77123 0 -2.5 Mabsadd L
.37429 .77123 -1.875 0 Mabsadd L closepath F
.37524 .77123 -1.875 0 Mabsadd m
.37524 .77123 0 2.5 Mabsadd L
.37524 .77123 1.875 0 Mabsadd L
.37524 .77123 0 -2.5 Mabsadd L
.37524 .77123 -1.875 0 Mabsadd L closepath F
.37619 .77123 -1.875 0 Mabsadd m
.37619 .77123 0 2.5 Mabsadd L
.37619 .77123 1.875 0 Mabsadd L
.37619 .77123 0 -2.5 Mabsadd L
.37619 .77123 -1.875 0 Mabsadd L closepath F
.37714 .77123 -1.875 0 Mabsadd m
.37714 .77123 0 2.5 Mabsadd L
.37714 .77123 1.875 0 Mabsadd L
.37714 .77123 0 -2.5 Mabsadd L
.37714 .77123 -1.875 0 Mabsadd L closepath F
.3781 .77123 -1.875 0 Mabsadd m
.3781 .77123 0 2.5 Mabsadd L
.3781 .77123 1.875 0 Mabsadd L
.3781 .77123 0 -2.5 Mabsadd L
.3781 .77123 -1.875 0 Mabsadd L closepath F
.37905 .77123 -1.875 0 Mabsadd m
.37905 .77123 0 2.5 Mabsadd L
.37905 .77123 1.875 0 Mabsadd L
.37905 .77123 0 -2.5 Mabsadd L
.37905 .77123 -1.875 0 Mabsadd L closepath F
.38 .77123 -1.875 0 Mabsadd m
.38 .77123 0 2.5 Mabsadd L
.38 .77123 1.875 0 Mabsadd L
.38 .77123 0 -2.5 Mabsadd L
.38 .77123 -1.875 0 Mabsadd L closepath F
.38095 .77123 -1.875 0 Mabsadd m
.38095 .77123 0 2.5 Mabsadd L
.38095 .77123 1.875 0 Mabsadd L
.38095 .77123 0 -2.5 Mabsadd L
.38095 .77123 -1.875 0 Mabsadd L closepath F
.3819 .77123 -1.875 0 Mabsadd m
.3819 .77123 0 2.5 Mabsadd L
.3819 .77123 1.875 0 Mabsadd L
.3819 .77123 0 -2.5 Mabsadd L
.3819 .77123 -1.875 0 Mabsadd L closepath F
.38286 .77123 -1.875 0 Mabsadd m
.38286 .77123 0 2.5 Mabsadd L
.38286 .77123 1.875 0 Mabsadd L
.38286 .77123 0 -2.5 Mabsadd L
.38286 .77123 -1.875 0 Mabsadd L closepath F
.38381 .77123 -1.875 0 Mabsadd m
.38381 .77123 0 2.5 Mabsadd L
.38381 .77123 1.875 0 Mabsadd L
.38381 .77123 0 -2.5 Mabsadd L
.38381 .77123 -1.875 0 Mabsadd L closepath F
.38476 .77123 -1.875 0 Mabsadd m
.38476 .77123 0 2.5 Mabsadd L
.38476 .77123 1.875 0 Mabsadd L
.38476 .77123 0 -2.5 Mabsadd L
.38476 .77123 -1.875 0 Mabsadd L closepath F
.38571 .77123 -1.875 0 Mabsadd m
.38571 .77123 0 2.5 Mabsadd L
.38571 .77123 1.875 0 Mabsadd L
.38571 .77123 0 -2.5 Mabsadd L
.38571 .77123 -1.875 0 Mabsadd L closepath F
.38667 .77123 -1.875 0 Mabsadd m
.38667 .77123 0 2.5 Mabsadd L
.38667 .77123 1.875 0 Mabsadd L
.38667 .77123 0 -2.5 Mabsadd L
.38667 .77123 -1.875 0 Mabsadd L closepath F
.38762 .77124 -1.875 0 Mabsadd m
.38762 .77124 0 2.5 Mabsadd L
.38762 .77124 1.875 0 Mabsadd L
.38762 .77124 0 -2.5 Mabsadd L
.38762 .77124 -1.875 0 Mabsadd L closepath F
.38857 .77123 -1.875 0 Mabsadd m
.38857 .77123 0 2.5 Mabsadd L
.38857 .77123 1.875 0 Mabsadd L
.38857 .77123 0 -2.5 Mabsadd L
.38857 .77123 -1.875 0 Mabsadd L closepath F
.38952 .77123 -1.875 0 Mabsadd m
.38952 .77123 0 2.5 Mabsadd L
.38952 .77123 1.875 0 Mabsadd L
.38952 .77123 0 -2.5 Mabsadd L
.38952 .77123 -1.875 0 Mabsadd L closepath F
.39048 .77122 -1.875 0 Mabsadd m
.39048 .77122 0 2.5 Mabsadd L
.39048 .77122 1.875 0 Mabsadd L
.39048 .77122 0 -2.5 Mabsadd L
.39048 .77122 -1.875 0 Mabsadd L closepath F
.39143 .7712 -1.875 0 Mabsadd m
.39143 .7712 0 2.5 Mabsadd L
.39143 .7712 1.875 0 Mabsadd L
.39143 .7712 0 -2.5 Mabsadd L
.39143 .7712 -1.875 0 Mabsadd L closepath F
.39238 .77113 -1.875 0 Mabsadd m
.39238 .77113 0 2.5 Mabsadd L
.39238 .77113 1.875 0 Mabsadd L
.39238 .77113 0 -2.5 Mabsadd L
.39238 .77113 -1.875 0 Mabsadd L closepath F
.39333 .77099 -1.875 0 Mabsadd m
.39333 .77099 0 2.5 Mabsadd L
.39333 .77099 1.875 0 Mabsadd L
.39333 .77099 0 -2.5 Mabsadd L
.39333 .77099 -1.875 0 Mabsadd L closepath F
.39429 .77076 -1.875 0 Mabsadd m
.39429 .77076 0 2.5 Mabsadd L
.39429 .77076 1.875 0 Mabsadd L
.39429 .77076 0 -2.5 Mabsadd L
.39429 .77076 -1.875 0 Mabsadd L closepath F
.39524 .76376 -1.875 0 Mabsadd m
.39524 .76376 0 2.5 Mabsadd L
.39524 .76376 1.875 0 Mabsadd L
.39524 .76376 0 -2.5 Mabsadd L
.39524 .76376 -1.875 0 Mabsadd L closepath F
.39619 .68249 -1.875 0 Mabsadd m
.39619 .68249 0 2.5 Mabsadd L
.39619 .68249 1.875 0 Mabsadd L
.39619 .68249 0 -2.5 Mabsadd L
.39619 .68249 -1.875 0 Mabsadd L closepath F
.39714 .7359 -1.875 0 Mabsadd m
.39714 .7359 0 2.5 Mabsadd L
.39714 .7359 1.875 0 Mabsadd L
.39714 .7359 0 -2.5 Mabsadd L
.39714 .7359 -1.875 0 Mabsadd L closepath F
.3981 .07073 -1.875 0 Mabsadd m
.3981 .07073 0 2.5 Mabsadd L
.3981 .07073 1.875 0 Mabsadd L
.3981 .07073 0 -2.5 Mabsadd L
.3981 .07073 -1.875 0 Mabsadd L closepath F
.39905 .64249 -1.875 0 Mabsadd m
.39905 .64249 0 2.5 Mabsadd L
.39905 .64249 1.875 0 Mabsadd L
.39905 .64249 0 -2.5 Mabsadd L
.39905 .64249 -1.875 0 Mabsadd L closepath F
.4 .64858 -1.875 0 Mabsadd m
.4 .64858 0 2.5 Mabsadd L
.4 .64858 1.875 0 Mabsadd L
.4 .64858 0 -2.5 Mabsadd L
.4 .64858 -1.875 0 Mabsadd L closepath F
.40095 .70888 -1.875 0 Mabsadd m
.40095 .70888 0 2.5 Mabsadd L
.40095 .70888 1.875 0 Mabsadd L
.40095 .70888 0 -2.5 Mabsadd L
.40095 .70888 -1.875 0 Mabsadd L closepath F
.4019 .71254 -1.875 0 Mabsadd m
.4019 .71254 0 2.5 Mabsadd L
.4019 .71254 1.875 0 Mabsadd L
.4019 .71254 0 -2.5 Mabsadd L
.4019 .71254 -1.875 0 Mabsadd L closepath F
.40286 .71272 -1.875 0 Mabsadd m
.40286 .71272 0 2.5 Mabsadd L
.40286 .71272 1.875 0 Mabsadd L
.40286 .71272 0 -2.5 Mabsadd L
.40286 .71272 -1.875 0 Mabsadd L closepath F
.40381 .71286 -1.875 0 Mabsadd m
.40381 .71286 0 2.5 Mabsadd L
.40381 .71286 1.875 0 Mabsadd L
.40381 .71286 0 -2.5 Mabsadd L
.40381 .71286 -1.875 0 Mabsadd L closepath F
.40476 .71291 -1.875 0 Mabsadd m
.40476 .71291 0 2.5 Mabsadd L
.40476 .71291 1.875 0 Mabsadd L
.40476 .71291 0 -2.5 Mabsadd L
.40476 .71291 -1.875 0 Mabsadd L closepath F
.40571 .71292 -1.875 0 Mabsadd m
.40571 .71292 0 2.5 Mabsadd L
.40571 .71292 1.875 0 Mabsadd L
.40571 .71292 0 -2.5 Mabsadd L
.40571 .71292 -1.875 0 Mabsadd L closepath F
.40667 .71293 -1.875 0 Mabsadd m
.40667 .71293 0 2.5 Mabsadd L
.40667 .71293 1.875 0 Mabsadd L
.40667 .71293 0 -2.5 Mabsadd L
.40667 .71293 -1.875 0 Mabsadd L closepath F
.40762 .71293 -1.875 0 Mabsadd m
.40762 .71293 0 2.5 Mabsadd L
.40762 .71293 1.875 0 Mabsadd L
.40762 .71293 0 -2.5 Mabsadd L
.40762 .71293 -1.875 0 Mabsadd L closepath F
.40857 .71293 -1.875 0 Mabsadd m
.40857 .71293 0 2.5 Mabsadd L
.40857 .71293 1.875 0 Mabsadd L
.40857 .71293 0 -2.5 Mabsadd L
.40857 .71293 -1.875 0 Mabsadd L closepath F
.40952 .71293 -1.875 0 Mabsadd m
.40952 .71293 0 2.5 Mabsadd L
.40952 .71293 1.875 0 Mabsadd L
.40952 .71293 0 -2.5 Mabsadd L
.40952 .71293 -1.875 0 Mabsadd L closepath F
.41048 .71293 -1.875 0 Mabsadd m
.41048 .71293 0 2.5 Mabsadd L
.41048 .71293 1.875 0 Mabsadd L
.41048 .71293 0 -2.5 Mabsadd L
.41048 .71293 -1.875 0 Mabsadd L closepath F
.41143 .71293 -1.875 0 Mabsadd m
.41143 .71293 0 2.5 Mabsadd L
.41143 .71293 1.875 0 Mabsadd L
.41143 .71293 0 -2.5 Mabsadd L
.41143 .71293 -1.875 0 Mabsadd L closepath F
.41238 .71293 -1.875 0 Mabsadd m
.41238 .71293 0 2.5 Mabsadd L
.41238 .71293 1.875 0 Mabsadd L
.41238 .71293 0 -2.5 Mabsadd L
.41238 .71293 -1.875 0 Mabsadd L closepath F
.41333 .71293 -1.875 0 Mabsadd m
.41333 .71293 0 2.5 Mabsadd L
.41333 .71293 1.875 0 Mabsadd L
.41333 .71293 0 -2.5 Mabsadd L
.41333 .71293 -1.875 0 Mabsadd L closepath F
.41429 .71293 -1.875 0 Mabsadd m
.41429 .71293 0 2.5 Mabsadd L
.41429 .71293 1.875 0 Mabsadd L
.41429 .71293 0 -2.5 Mabsadd L
.41429 .71293 -1.875 0 Mabsadd L closepath F
.41524 .71293 -1.875 0 Mabsadd m
.41524 .71293 0 2.5 Mabsadd L
.41524 .71293 1.875 0 Mabsadd L
.41524 .71293 0 -2.5 Mabsadd L
.41524 .71293 -1.875 0 Mabsadd L closepath F
.41619 .71293 -1.875 0 Mabsadd m
.41619 .71293 0 2.5 Mabsadd L
.41619 .71293 1.875 0 Mabsadd L
.41619 .71293 0 -2.5 Mabsadd L
.41619 .71293 -1.875 0 Mabsadd L closepath F
.41714 .71293 -1.875 0 Mabsadd m
.41714 .71293 0 2.5 Mabsadd L
.41714 .71293 1.875 0 Mabsadd L
.41714 .71293 0 -2.5 Mabsadd L
.41714 .71293 -1.875 0 Mabsadd L closepath F
.4181 .71293 -1.875 0 Mabsadd m
.4181 .71293 0 2.5 Mabsadd L
.4181 .71293 1.875 0 Mabsadd L
.4181 .71293 0 -2.5 Mabsadd L
.4181 .71293 -1.875 0 Mabsadd L closepath F
.41905 .71293 -1.875 0 Mabsadd m
.41905 .71293 0 2.5 Mabsadd L
.41905 .71293 1.875 0 Mabsadd L
.41905 .71293 0 -2.5 Mabsadd L
.41905 .71293 -1.875 0 Mabsadd L closepath F
.42 .71293 -1.875 0 Mabsadd m
.42 .71293 0 2.5 Mabsadd L
.42 .71293 1.875 0 Mabsadd L
.42 .71293 0 -2.5 Mabsadd L
.42 .71293 -1.875 0 Mabsadd L closepath F
.42095 .71293 -1.875 0 Mabsadd m
.42095 .71293 0 2.5 Mabsadd L
.42095 .71293 1.875 0 Mabsadd L
.42095 .71293 0 -2.5 Mabsadd L
.42095 .71293 -1.875 0 Mabsadd L closepath F
.4219 .71293 -1.875 0 Mabsadd m
.4219 .71293 0 2.5 Mabsadd L
.4219 .71293 1.875 0 Mabsadd L
.4219 .71293 0 -2.5 Mabsadd L
.4219 .71293 -1.875 0 Mabsadd L closepath F
.42286 .71293 -1.875 0 Mabsadd m
.42286 .71293 0 2.5 Mabsadd L
.42286 .71293 1.875 0 Mabsadd L
.42286 .71293 0 -2.5 Mabsadd L
.42286 .71293 -1.875 0 Mabsadd L closepath F
.42381 .71293 -1.875 0 Mabsadd m
.42381 .71293 0 2.5 Mabsadd L
.42381 .71293 1.875 0 Mabsadd L
.42381 .71293 0 -2.5 Mabsadd L
.42381 .71293 -1.875 0 Mabsadd L closepath F
.42476 .71293 -1.875 0 Mabsadd m
.42476 .71293 0 2.5 Mabsadd L
.42476 .71293 1.875 0 Mabsadd L
.42476 .71293 0 -2.5 Mabsadd L
.42476 .71293 -1.875 0 Mabsadd L closepath F
.42571 .71293 -1.875 0 Mabsadd m
.42571 .71293 0 2.5 Mabsadd L
.42571 .71293 1.875 0 Mabsadd L
.42571 .71293 0 -2.5 Mabsadd L
.42571 .71293 -1.875 0 Mabsadd L closepath F
.42667 .71293 -1.875 0 Mabsadd m
.42667 .71293 0 2.5 Mabsadd L
.42667 .71293 1.875 0 Mabsadd L
.42667 .71293 0 -2.5 Mabsadd L
.42667 .71293 -1.875 0 Mabsadd L closepath F
.42762 .71293 -1.875 0 Mabsadd m
.42762 .71293 0 2.5 Mabsadd L
.42762 .71293 1.875 0 Mabsadd L
.42762 .71293 0 -2.5 Mabsadd L
.42762 .71293 -1.875 0 Mabsadd L closepath F
.42857 .71293 -1.875 0 Mabsadd m
.42857 .71293 0 2.5 Mabsadd L
.42857 .71293 1.875 0 Mabsadd L
.42857 .71293 0 -2.5 Mabsadd L
.42857 .71293 -1.875 0 Mabsadd L closepath F
.42952 .71293 -1.875 0 Mabsadd m
.42952 .71293 0 2.5 Mabsadd L
.42952 .71293 1.875 0 Mabsadd L
.42952 .71293 0 -2.5 Mabsadd L
.42952 .71293 -1.875 0 Mabsadd L closepath F
.43048 .71293 -1.875 0 Mabsadd m
.43048 .71293 0 2.5 Mabsadd L
.43048 .71293 1.875 0 Mabsadd L
.43048 .71293 0 -2.5 Mabsadd L
.43048 .71293 -1.875 0 Mabsadd L closepath F
.43143 .71293 -1.875 0 Mabsadd m
.43143 .71293 0 2.5 Mabsadd L
.43143 .71293 1.875 0 Mabsadd L
.43143 .71293 0 -2.5 Mabsadd L
.43143 .71293 -1.875 0 Mabsadd L closepath F
.43238 .71293 -1.875 0 Mabsadd m
.43238 .71293 0 2.5 Mabsadd L
.43238 .71293 1.875 0 Mabsadd L
.43238 .71293 0 -2.5 Mabsadd L
.43238 .71293 -1.875 0 Mabsadd L closepath F
.43333 .71293 -1.875 0 Mabsadd m
.43333 .71293 0 2.5 Mabsadd L
.43333 .71293 1.875 0 Mabsadd L
.43333 .71293 0 -2.5 Mabsadd L
.43333 .71293 -1.875 0 Mabsadd L closepath F
.43429 .71293 -1.875 0 Mabsadd m
.43429 .71293 0 2.5 Mabsadd L
.43429 .71293 1.875 0 Mabsadd L
.43429 .71293 0 -2.5 Mabsadd L
.43429 .71293 -1.875 0 Mabsadd L closepath F
.43524 .71293 -1.875 0 Mabsadd m
.43524 .71293 0 2.5 Mabsadd L
.43524 .71293 1.875 0 Mabsadd L
.43524 .71293 0 -2.5 Mabsadd L
.43524 .71293 -1.875 0 Mabsadd L closepath F
.43619 .71293 -1.875 0 Mabsadd m
.43619 .71293 0 2.5 Mabsadd L
.43619 .71293 1.875 0 Mabsadd L
.43619 .71293 0 -2.5 Mabsadd L
.43619 .71293 -1.875 0 Mabsadd L closepath F
.43714 .71293 -1.875 0 Mabsadd m
.43714 .71293 0 2.5 Mabsadd L
.43714 .71293 1.875 0 Mabsadd L
.43714 .71293 0 -2.5 Mabsadd L
.43714 .71293 -1.875 0 Mabsadd L closepath F
.4381 .71293 -1.875 0 Mabsadd m
.4381 .71293 0 2.5 Mabsadd L
.4381 .71293 1.875 0 Mabsadd L
.4381 .71293 0 -2.5 Mabsadd L
.4381 .71293 -1.875 0 Mabsadd L closepath F
.43905 .71293 -1.875 0 Mabsadd m
.43905 .71293 0 2.5 Mabsadd L
.43905 .71293 1.875 0 Mabsadd L
.43905 .71293 0 -2.5 Mabsadd L
.43905 .71293 -1.875 0 Mabsadd L closepath F
.44 .71293 -1.875 0 Mabsadd m
.44 .71293 0 2.5 Mabsadd L
.44 .71293 1.875 0 Mabsadd L
.44 .71293 0 -2.5 Mabsadd L
.44 .71293 -1.875 0 Mabsadd L closepath F
.44095 .71293 -1.875 0 Mabsadd m
.44095 .71293 0 2.5 Mabsadd L
.44095 .71293 1.875 0 Mabsadd L
.44095 .71293 0 -2.5 Mabsadd L
.44095 .71293 -1.875 0 Mabsadd L closepath F
.4419 .71293 -1.875 0 Mabsadd m
.4419 .71293 0 2.5 Mabsadd L
.4419 .71293 1.875 0 Mabsadd L
.4419 .71293 0 -2.5 Mabsadd L
.4419 .71293 -1.875 0 Mabsadd L closepath F
.44286 .71293 -1.875 0 Mabsadd m
.44286 .71293 0 2.5 Mabsadd L
.44286 .71293 1.875 0 Mabsadd L
.44286 .71293 0 -2.5 Mabsadd L
.44286 .71293 -1.875 0 Mabsadd L closepath F
.44381 .71293 -1.875 0 Mabsadd m
.44381 .71293 0 2.5 Mabsadd L
.44381 .71293 1.875 0 Mabsadd L
.44381 .71293 0 -2.5 Mabsadd L
.44381 .71293 -1.875 0 Mabsadd L closepath F
.44476 .71291 -1.875 0 Mabsadd m
.44476 .71291 0 2.5 Mabsadd L
.44476 .71291 1.875 0 Mabsadd L
.44476 .71291 0 -2.5 Mabsadd L
.44476 .71291 -1.875 0 Mabsadd L closepath F
.44571 .71285 -1.875 0 Mabsadd m
.44571 .71285 0 2.5 Mabsadd L
.44571 .71285 1.875 0 Mabsadd L
.44571 .71285 0 -2.5 Mabsadd L
.44571 .71285 -1.875 0 Mabsadd L closepath F
.44667 .71271 -1.875 0 Mabsadd m
.44667 .71271 0 2.5 Mabsadd L
.44667 .71271 1.875 0 Mabsadd L
.44667 .71271 0 -2.5 Mabsadd L
.44667 .71271 -1.875 0 Mabsadd L closepath F
.44762 .71252 -1.875 0 Mabsadd m
.44762 .71252 0 2.5 Mabsadd L
.44762 .71252 1.875 0 Mabsadd L
.44762 .71252 0 -2.5 Mabsadd L
.44762 .71252 -1.875 0 Mabsadd L closepath F
.44857 .70803 -1.875 0 Mabsadd m
.44857 .70803 0 2.5 Mabsadd L
.44857 .70803 1.875 0 Mabsadd L
.44857 .70803 0 -2.5 Mabsadd L
.44857 .70803 -1.875 0 Mabsadd L closepath F
.44952 .64125 -1.875 0 Mabsadd m
.44952 .64125 0 2.5 Mabsadd L
.44952 .64125 1.875 0 Mabsadd L
.44952 .64125 0 -2.5 Mabsadd L
.44952 .64125 -1.875 0 Mabsadd L closepath F
.45048 .65509 -1.875 0 Mabsadd m
.45048 .65509 0 2.5 Mabsadd L
.45048 .65509 1.875 0 Mabsadd L
.45048 .65509 0 -2.5 Mabsadd L
.45048 .65509 -1.875 0 Mabsadd L closepath F
.45143 .03084 -1.875 0 Mabsadd m
.45143 .03084 0 2.5 Mabsadd L
.45143 .03084 1.875 0 Mabsadd L
.45143 .03084 0 -2.5 Mabsadd L
.45143 .03084 -1.875 0 Mabsadd L closepath F
.45238 .69282 -1.875 0 Mabsadd m
.45238 .69282 0 2.5 Mabsadd L
.45238 .69282 1.875 0 Mabsadd L
.45238 .69282 0 -2.5 Mabsadd L
.45238 .69282 -1.875 0 Mabsadd L closepath F
.45333 .65537 -1.875 0 Mabsadd m
.45333 .65537 0 2.5 Mabsadd L
.45333 .65537 1.875 0 Mabsadd L
.45333 .65537 0 -2.5 Mabsadd L
.45333 .65537 -1.875 0 Mabsadd L closepath F
.45429 .7303 -1.875 0 Mabsadd m
.45429 .7303 0 2.5 Mabsadd L
.45429 .7303 1.875 0 Mabsadd L
.45429 .7303 0 -2.5 Mabsadd L
.45429 .7303 -1.875 0 Mabsadd L closepath F
.45524 .73606 -1.875 0 Mabsadd m
.45524 .73606 0 2.5 Mabsadd L
.45524 .73606 1.875 0 Mabsadd L
.45524 .73606 0 -2.5 Mabsadd L
.45524 .73606 -1.875 0 Mabsadd L closepath F
.45619 .73627 -1.875 0 Mabsadd m
.45619 .73627 0 2.5 Mabsadd L
.45619 .73627 1.875 0 Mabsadd L
.45619 .73627 0 -2.5 Mabsadd L
.45619 .73627 -1.875 0 Mabsadd L closepath F
.45714 .73641 -1.875 0 Mabsadd m
.45714 .73641 0 2.5 Mabsadd L
.45714 .73641 1.875 0 Mabsadd L
.45714 .73641 0 -2.5 Mabsadd L
.45714 .73641 -1.875 0 Mabsadd L closepath F
.4581 .73647 -1.875 0 Mabsadd m
.4581 .73647 0 2.5 Mabsadd L
.4581 .73647 1.875 0 Mabsadd L
.4581 .73647 0 -2.5 Mabsadd L
.4581 .73647 -1.875 0 Mabsadd L closepath F
.45905 .73649 -1.875 0 Mabsadd m
.45905 .73649 0 2.5 Mabsadd L
.45905 .73649 1.875 0 Mabsadd L
.45905 .73649 0 -2.5 Mabsadd L
.45905 .73649 -1.875 0 Mabsadd L closepath F
.46 .7365 -1.875 0 Mabsadd m
.46 .7365 0 2.5 Mabsadd L
.46 .7365 1.875 0 Mabsadd L
.46 .7365 0 -2.5 Mabsadd L
.46 .7365 -1.875 0 Mabsadd L closepath F
.46095 .73651 -1.875 0 Mabsadd m
.46095 .73651 0 2.5 Mabsadd L
.46095 .73651 1.875 0 Mabsadd L
.46095 .73651 0 -2.5 Mabsadd L
.46095 .73651 -1.875 0 Mabsadd L closepath F
.4619 .73651 -1.875 0 Mabsadd m
.4619 .73651 0 2.5 Mabsadd L
.4619 .73651 1.875 0 Mabsadd L
.4619 .73651 0 -2.5 Mabsadd L
.4619 .73651 -1.875 0 Mabsadd L closepath F
.46286 .73651 -1.875 0 Mabsadd m
.46286 .73651 0 2.5 Mabsadd L
.46286 .73651 1.875 0 Mabsadd L
.46286 .73651 0 -2.5 Mabsadd L
.46286 .73651 -1.875 0 Mabsadd L closepath F
.46381 .73651 -1.875 0 Mabsadd m
.46381 .73651 0 2.5 Mabsadd L
.46381 .73651 1.875 0 Mabsadd L
.46381 .73651 0 -2.5 Mabsadd L
.46381 .73651 -1.875 0 Mabsadd L closepath F
.46476 .7365 -1.875 0 Mabsadd m
.46476 .7365 0 2.5 Mabsadd L
.46476 .7365 1.875 0 Mabsadd L
.46476 .7365 0 -2.5 Mabsadd L
.46476 .7365 -1.875 0 Mabsadd L closepath F
.46571 .7365 -1.875 0 Mabsadd m
.46571 .7365 0 2.5 Mabsadd L
.46571 .7365 1.875 0 Mabsadd L
.46571 .7365 0 -2.5 Mabsadd L
.46571 .7365 -1.875 0 Mabsadd L closepath F
.46667 .73651 -1.875 0 Mabsadd m
.46667 .73651 0 2.5 Mabsadd L
.46667 .73651 1.875 0 Mabsadd L
.46667 .73651 0 -2.5 Mabsadd L
.46667 .73651 -1.875 0 Mabsadd L closepath F
.46762 .7365 -1.875 0 Mabsadd m
.46762 .7365 0 2.5 Mabsadd L
.46762 .7365 1.875 0 Mabsadd L
.46762 .7365 0 -2.5 Mabsadd L
.46762 .7365 -1.875 0 Mabsadd L closepath F
.46857 .7365 -1.875 0 Mabsadd m
.46857 .7365 0 2.5 Mabsadd L
.46857 .7365 1.875 0 Mabsadd L
.46857 .7365 0 -2.5 Mabsadd L
.46857 .7365 -1.875 0 Mabsadd L closepath F
.46952 .7365 -1.875 0 Mabsadd m
.46952 .7365 0 2.5 Mabsadd L
.46952 .7365 1.875 0 Mabsadd L
.46952 .7365 0 -2.5 Mabsadd L
.46952 .7365 -1.875 0 Mabsadd L closepath F
.47048 .7365 -1.875 0 Mabsadd m
.47048 .7365 0 2.5 Mabsadd L
.47048 .7365 1.875 0 Mabsadd L
.47048 .7365 0 -2.5 Mabsadd L
.47048 .7365 -1.875 0 Mabsadd L closepath F
.47143 .73651 -1.875 0 Mabsadd m
.47143 .73651 0 2.5 Mabsadd L
.47143 .73651 1.875 0 Mabsadd L
.47143 .73651 0 -2.5 Mabsadd L
.47143 .73651 -1.875 0 Mabsadd L closepath F
.47238 .73651 -1.875 0 Mabsadd m
.47238 .73651 0 2.5 Mabsadd L
.47238 .73651 1.875 0 Mabsadd L
.47238 .73651 0 -2.5 Mabsadd L
.47238 .73651 -1.875 0 Mabsadd L closepath F
.47333 .7365 -1.875 0 Mabsadd m
.47333 .7365 0 2.5 Mabsadd L
.47333 .7365 1.875 0 Mabsadd L
.47333 .7365 0 -2.5 Mabsadd L
.47333 .7365 -1.875 0 Mabsadd L closepath F
.47429 .73651 -1.875 0 Mabsadd m
.47429 .73651 0 2.5 Mabsadd L
.47429 .73651 1.875 0 Mabsadd L
.47429 .73651 0 -2.5 Mabsadd L
.47429 .73651 -1.875 0 Mabsadd L closepath F
.47524 .7365 -1.875 0 Mabsadd m
.47524 .7365 0 2.5 Mabsadd L
.47524 .7365 1.875 0 Mabsadd L
.47524 .7365 0 -2.5 Mabsadd L
.47524 .7365 -1.875 0 Mabsadd L closepath F
.47619 .7365 -1.875 0 Mabsadd m
.47619 .7365 0 2.5 Mabsadd L
.47619 .7365 1.875 0 Mabsadd L
.47619 .7365 0 -2.5 Mabsadd L
.47619 .7365 -1.875 0 Mabsadd L closepath F
.47714 .7365 -1.875 0 Mabsadd m
.47714 .7365 0 2.5 Mabsadd L
.47714 .7365 1.875 0 Mabsadd L
.47714 .7365 0 -2.5 Mabsadd L
.47714 .7365 -1.875 0 Mabsadd L closepath F
.4781 .73651 -1.875 0 Mabsadd m
.4781 .73651 0 2.5 Mabsadd L
.4781 .73651 1.875 0 Mabsadd L
.4781 .73651 0 -2.5 Mabsadd L
.4781 .73651 -1.875 0 Mabsadd L closepath F
.47905 .7365 -1.875 0 Mabsadd m
.47905 .7365 0 2.5 Mabsadd L
.47905 .7365 1.875 0 Mabsadd L
.47905 .7365 0 -2.5 Mabsadd L
.47905 .7365 -1.875 0 Mabsadd L closepath F
.48 .7365 -1.875 0 Mabsadd m
.48 .7365 0 2.5 Mabsadd L
.48 .7365 1.875 0 Mabsadd L
.48 .7365 0 -2.5 Mabsadd L
.48 .7365 -1.875 0 Mabsadd L closepath F
.48095 .73651 -1.875 0 Mabsadd m
.48095 .73651 0 2.5 Mabsadd L
.48095 .73651 1.875 0 Mabsadd L
.48095 .73651 0 -2.5 Mabsadd L
.48095 .73651 -1.875 0 Mabsadd L closepath F
.4819 .7365 -1.875 0 Mabsadd m
.4819 .7365 0 2.5 Mabsadd L
.4819 .7365 1.875 0 Mabsadd L
.4819 .7365 0 -2.5 Mabsadd L
.4819 .7365 -1.875 0 Mabsadd L closepath F
.48286 .7365 -1.875 0 Mabsadd m
.48286 .7365 0 2.5 Mabsadd L
.48286 .7365 1.875 0 Mabsadd L
.48286 .7365 0 -2.5 Mabsadd L
.48286 .7365 -1.875 0 Mabsadd L closepath F
.48381 .7365 -1.875 0 Mabsadd m
.48381 .7365 0 2.5 Mabsadd L
.48381 .7365 1.875 0 Mabsadd L
.48381 .7365 0 -2.5 Mabsadd L
.48381 .7365 -1.875 0 Mabsadd L closepath F
.48476 .73651 -1.875 0 Mabsadd m
.48476 .73651 0 2.5 Mabsadd L
.48476 .73651 1.875 0 Mabsadd L
.48476 .73651 0 -2.5 Mabsadd L
.48476 .73651 -1.875 0 Mabsadd L closepath F
.48571 .73651 -1.875 0 Mabsadd m
.48571 .73651 0 2.5 Mabsadd L
.48571 .73651 1.875 0 Mabsadd L
.48571 .73651 0 -2.5 Mabsadd L
.48571 .73651 -1.875 0 Mabsadd L closepath F
.48667 .7365 -1.875 0 Mabsadd m
.48667 .7365 0 2.5 Mabsadd L
.48667 .7365 1.875 0 Mabsadd L
.48667 .7365 0 -2.5 Mabsadd L
.48667 .7365 -1.875 0 Mabsadd L closepath F
.48762 .73651 -1.875 0 Mabsadd m
.48762 .73651 0 2.5 Mabsadd L
.48762 .73651 1.875 0 Mabsadd L
.48762 .73651 0 -2.5 Mabsadd L
.48762 .73651 -1.875 0 Mabsadd L closepath F
.48857 .73651 -1.875 0 Mabsadd m
.48857 .73651 0 2.5 Mabsadd L
.48857 .73651 1.875 0 Mabsadd L
.48857 .73651 0 -2.5 Mabsadd L
.48857 .73651 -1.875 0 Mabsadd L closepath F
.48952 .7365 -1.875 0 Mabsadd m
.48952 .7365 0 2.5 Mabsadd L
.48952 .7365 1.875 0 Mabsadd L
.48952 .7365 0 -2.5 Mabsadd L
.48952 .7365 -1.875 0 Mabsadd L closepath F
.49048 .73651 -1.875 0 Mabsadd m
.49048 .73651 0 2.5 Mabsadd L
.49048 .73651 1.875 0 Mabsadd L
.49048 .73651 0 -2.5 Mabsadd L
.49048 .73651 -1.875 0 Mabsadd L closepath F
.49143 .73651 -1.875 0 Mabsadd m
.49143 .73651 0 2.5 Mabsadd L
.49143 .73651 1.875 0 Mabsadd L
.49143 .73651 0 -2.5 Mabsadd L
.49143 .73651 -1.875 0 Mabsadd L closepath F
.49238 .73651 -1.875 0 Mabsadd m
.49238 .73651 0 2.5 Mabsadd L
.49238 .73651 1.875 0 Mabsadd L
.49238 .73651 0 -2.5 Mabsadd L
.49238 .73651 -1.875 0 Mabsadd L closepath F
.49333 .7365 -1.875 0 Mabsadd m
.49333 .7365 0 2.5 Mabsadd L
.49333 .7365 1.875 0 Mabsadd L
.49333 .7365 0 -2.5 Mabsadd L
.49333 .7365 -1.875 0 Mabsadd L closepath F
.49429 .73651 -1.875 0 Mabsadd m
.49429 .73651 0 2.5 Mabsadd L
.49429 .73651 1.875 0 Mabsadd L
.49429 .73651 0 -2.5 Mabsadd L
.49429 .73651 -1.875 0 Mabsadd L closepath F
.49524 .73651 -1.875 0 Mabsadd m
.49524 .73651 0 2.5 Mabsadd L
.49524 .73651 1.875 0 Mabsadd L
.49524 .73651 0 -2.5 Mabsadd L
.49524 .73651 -1.875 0 Mabsadd L closepath F
.49619 .7365 -1.875 0 Mabsadd m
.49619 .7365 0 2.5 Mabsadd L
.49619 .7365 1.875 0 Mabsadd L
.49619 .7365 0 -2.5 Mabsadd L
.49619 .7365 -1.875 0 Mabsadd L closepath F
.49714 .7365 -1.875 0 Mabsadd m
.49714 .7365 0 2.5 Mabsadd L
.49714 .7365 1.875 0 Mabsadd L
.49714 .7365 0 -2.5 Mabsadd L
.49714 .7365 -1.875 0 Mabsadd L closepath F
.4981 .73648 -1.875 0 Mabsadd m
.4981 .73648 0 2.5 Mabsadd L
.4981 .73648 1.875 0 Mabsadd L
.4981 .73648 0 -2.5 Mabsadd L
.4981 .73648 -1.875 0 Mabsadd L closepath F
.49905 .73643 -1.875 0 Mabsadd m
.49905 .73643 0 2.5 Mabsadd L
.49905 .73643 1.875 0 Mabsadd L
.49905 .73643 0 -2.5 Mabsadd L
.49905 .73643 -1.875 0 Mabsadd L closepath F
.5 .73631 -1.875 0 Mabsadd m
.5 .73631 0 2.5 Mabsadd L
.5 .73631 1.875 0 Mabsadd L
.5 .73631 0 -2.5 Mabsadd L
.5 .73631 -1.875 0 Mabsadd L closepath F
.50095 .73613 -1.875 0 Mabsadd m
.50095 .73613 0 2.5 Mabsadd L
.50095 .73613 1.875 0 Mabsadd L
.50095 .73613 0 -2.5 Mabsadd L
.50095 .73613 -1.875 0 Mabsadd L closepath F
.5019 .73332 -1.875 0 Mabsadd m
.5019 .73332 0 2.5 Mabsadd L
.5019 .73332 1.875 0 Mabsadd L
.5019 .73332 0 -2.5 Mabsadd L
.5019 .73332 -1.875 0 Mabsadd L closepath F
.50286 .68083 -1.875 0 Mabsadd m
.50286 .68083 0 2.5 Mabsadd L
.50286 .68083 1.875 0 Mabsadd L
.50286 .68083 0 -2.5 Mabsadd L
.50286 .68083 -1.875 0 Mabsadd L closepath F
.50381 .65046 -1.875 0 Mabsadd m
.50381 .65046 0 2.5 Mabsadd L
.50381 .65046 1.875 0 Mabsadd L
.50381 .65046 0 -2.5 Mabsadd L
.50381 .65046 -1.875 0 Mabsadd L closepath F
.50476 .16472 -1.875 0 Mabsadd m
.50476 .16472 0 2.5 Mabsadd L
.50476 .16472 1.875 0 Mabsadd L
.50476 .16472 0 -2.5 Mabsadd L
.50476 .16472 -1.875 0 Mabsadd L closepath F
.50571 .79618 -1.875 0 Mabsadd m
.50571 .79618 0 2.5 Mabsadd L
.50571 .79618 1.875 0 Mabsadd L
.50571 .79618 0 -2.5 Mabsadd L
.50571 .79618 -1.875 0 Mabsadd L closepath F
.50667 .72917 -1.875 0 Mabsadd m
.50667 .72917 0 2.5 Mabsadd L
.50667 .72917 1.875 0 Mabsadd L
.50667 .72917 0 -2.5 Mabsadd L
.50667 .72917 -1.875 0 Mabsadd L closepath F
.50762 .8178 -1.875 0 Mabsadd m
.50762 .8178 0 2.5 Mabsadd L
.50762 .8178 1.875 0 Mabsadd L
.50762 .8178 0 -2.5 Mabsadd L
.50762 .8178 -1.875 0 Mabsadd L closepath F
.50857 .82665 -1.875 0 Mabsadd m
.50857 .82665 0 2.5 Mabsadd L
.50857 .82665 1.875 0 Mabsadd L
.50857 .82665 0 -2.5 Mabsadd L
.50857 .82665 -1.875 0 Mabsadd L closepath F
.50952 .82692 -1.875 0 Mabsadd m
.50952 .82692 0 2.5 Mabsadd L
.50952 .82692 1.875 0 Mabsadd L
.50952 .82692 0 -2.5 Mabsadd L
.50952 .82692 -1.875 0 Mabsadd L closepath F
.51048 .82707 -1.875 0 Mabsadd m
.51048 .82707 0 2.5 Mabsadd L
.51048 .82707 1.875 0 Mabsadd L
.51048 .82707 0 -2.5 Mabsadd L
.51048 .82707 -1.875 0 Mabsadd L closepath F
.51143 .82714 -1.875 0 Mabsadd m
.51143 .82714 0 2.5 Mabsadd L
.51143 .82714 1.875 0 Mabsadd L
.51143 .82714 0 -2.5 Mabsadd L
.51143 .82714 -1.875 0 Mabsadd L closepath F
.51238 .82717 -1.875 0 Mabsadd m
.51238 .82717 0 2.5 Mabsadd L
.51238 .82717 1.875 0 Mabsadd L
.51238 .82717 0 -2.5 Mabsadd L
.51238 .82717 -1.875 0 Mabsadd L closepath F
.51333 .82717 -1.875 0 Mabsadd m
.51333 .82717 0 2.5 Mabsadd L
.51333 .82717 1.875 0 Mabsadd L
.51333 .82717 0 -2.5 Mabsadd L
.51333 .82717 -1.875 0 Mabsadd L closepath F
.51429 .82718 -1.875 0 Mabsadd m
.51429 .82718 0 2.5 Mabsadd L
.51429 .82718 1.875 0 Mabsadd L
.51429 .82718 0 -2.5 Mabsadd L
.51429 .82718 -1.875 0 Mabsadd L closepath F
.51524 .82718 -1.875 0 Mabsadd m
.51524 .82718 0 2.5 Mabsadd L
.51524 .82718 1.875 0 Mabsadd L
.51524 .82718 0 -2.5 Mabsadd L
.51524 .82718 -1.875 0 Mabsadd L closepath F
.51619 .82718 -1.875 0 Mabsadd m
.51619 .82718 0 2.5 Mabsadd L
.51619 .82718 1.875 0 Mabsadd L
.51619 .82718 0 -2.5 Mabsadd L
.51619 .82718 -1.875 0 Mabsadd L closepath F
.51714 .82718 -1.875 0 Mabsadd m
.51714 .82718 0 2.5 Mabsadd L
.51714 .82718 1.875 0 Mabsadd L
.51714 .82718 0 -2.5 Mabsadd L
.51714 .82718 -1.875 0 Mabsadd L closepath F
.5181 .82718 -1.875 0 Mabsadd m
.5181 .82718 0 2.5 Mabsadd L
.5181 .82718 1.875 0 Mabsadd L
.5181 .82718 0 -2.5 Mabsadd L
.5181 .82718 -1.875 0 Mabsadd L closepath F
.51905 .82718 -1.875 0 Mabsadd m
.51905 .82718 0 2.5 Mabsadd L
.51905 .82718 1.875 0 Mabsadd L
.51905 .82718 0 -2.5 Mabsadd L
.51905 .82718 -1.875 0 Mabsadd L closepath F
.52 .82718 -1.875 0 Mabsadd m
.52 .82718 0 2.5 Mabsadd L
.52 .82718 1.875 0 Mabsadd L
.52 .82718 0 -2.5 Mabsadd L
.52 .82718 -1.875 0 Mabsadd L closepath F
.52095 .82718 -1.875 0 Mabsadd m
.52095 .82718 0 2.5 Mabsadd L
.52095 .82718 1.875 0 Mabsadd L
.52095 .82718 0 -2.5 Mabsadd L
.52095 .82718 -1.875 0 Mabsadd L closepath F
.5219 .82718 -1.875 0 Mabsadd m
.5219 .82718 0 2.5 Mabsadd L
.5219 .82718 1.875 0 Mabsadd L
.5219 .82718 0 -2.5 Mabsadd L
.5219 .82718 -1.875 0 Mabsadd L closepath F
.52286 .82718 -1.875 0 Mabsadd m
.52286 .82718 0 2.5 Mabsadd L
.52286 .82718 1.875 0 Mabsadd L
.52286 .82718 0 -2.5 Mabsadd L
.52286 .82718 -1.875 0 Mabsadd L closepath F
.52381 .82718 -1.875 0 Mabsadd m
.52381 .82718 0 2.5 Mabsadd L
.52381 .82718 1.875 0 Mabsadd L
.52381 .82718 0 -2.5 Mabsadd L
.52381 .82718 -1.875 0 Mabsadd L closepath F
.52476 .82718 -1.875 0 Mabsadd m
.52476 .82718 0 2.5 Mabsadd L
.52476 .82718 1.875 0 Mabsadd L
.52476 .82718 0 -2.5 Mabsadd L
.52476 .82718 -1.875 0 Mabsadd L closepath F
.52571 .82718 -1.875 0 Mabsadd m
.52571 .82718 0 2.5 Mabsadd L
.52571 .82718 1.875 0 Mabsadd L
.52571 .82718 0 -2.5 Mabsadd L
.52571 .82718 -1.875 0 Mabsadd L closepath F
.52667 .82718 -1.875 0 Mabsadd m
.52667 .82718 0 2.5 Mabsadd L
.52667 .82718 1.875 0 Mabsadd L
.52667 .82718 0 -2.5 Mabsadd L
.52667 .82718 -1.875 0 Mabsadd L closepath F
.52762 .82718 -1.875 0 Mabsadd m
.52762 .82718 0 2.5 Mabsadd L
.52762 .82718 1.875 0 Mabsadd L
.52762 .82718 0 -2.5 Mabsadd L
.52762 .82718 -1.875 0 Mabsadd L closepath F
.52857 .82718 -1.875 0 Mabsadd m
.52857 .82718 0 2.5 Mabsadd L
.52857 .82718 1.875 0 Mabsadd L
.52857 .82718 0 -2.5 Mabsadd L
.52857 .82718 -1.875 0 Mabsadd L closepath F
.52952 .82718 -1.875 0 Mabsadd m
.52952 .82718 0 2.5 Mabsadd L
.52952 .82718 1.875 0 Mabsadd L
.52952 .82718 0 -2.5 Mabsadd L
.52952 .82718 -1.875 0 Mabsadd L closepath F
.53048 .82718 -1.875 0 Mabsadd m
.53048 .82718 0 2.5 Mabsadd L
.53048 .82718 1.875 0 Mabsadd L
.53048 .82718 0 -2.5 Mabsadd L
.53048 .82718 -1.875 0 Mabsadd L closepath F
.53143 .82718 -1.875 0 Mabsadd m
.53143 .82718 0 2.5 Mabsadd L
.53143 .82718 1.875 0 Mabsadd L
.53143 .82718 0 -2.5 Mabsadd L
.53143 .82718 -1.875 0 Mabsadd L closepath F
.53238 .82718 -1.875 0 Mabsadd m
.53238 .82718 0 2.5 Mabsadd L
.53238 .82718 1.875 0 Mabsadd L
.53238 .82718 0 -2.5 Mabsadd L
.53238 .82718 -1.875 0 Mabsadd L closepath F
.53333 .82718 -1.875 0 Mabsadd m
.53333 .82718 0 2.5 Mabsadd L
.53333 .82718 1.875 0 Mabsadd L
.53333 .82718 0 -2.5 Mabsadd L
.53333 .82718 -1.875 0 Mabsadd L closepath F
.53429 .82718 -1.875 0 Mabsadd m
.53429 .82718 0 2.5 Mabsadd L
.53429 .82718 1.875 0 Mabsadd L
.53429 .82718 0 -2.5 Mabsadd L
.53429 .82718 -1.875 0 Mabsadd L closepath F
.53524 .82718 -1.875 0 Mabsadd m
.53524 .82718 0 2.5 Mabsadd L
.53524 .82718 1.875 0 Mabsadd L
.53524 .82718 0 -2.5 Mabsadd L
.53524 .82718 -1.875 0 Mabsadd L closepath F
.53619 .82718 -1.875 0 Mabsadd m
.53619 .82718 0 2.5 Mabsadd L
.53619 .82718 1.875 0 Mabsadd L
.53619 .82718 0 -2.5 Mabsadd L
.53619 .82718 -1.875 0 Mabsadd L closepath F
.53714 .82718 -1.875 0 Mabsadd m
.53714 .82718 0 2.5 Mabsadd L
.53714 .82718 1.875 0 Mabsadd L
.53714 .82718 0 -2.5 Mabsadd L
.53714 .82718 -1.875 0 Mabsadd L closepath F
.5381 .82718 -1.875 0 Mabsadd m
.5381 .82718 0 2.5 Mabsadd L
.5381 .82718 1.875 0 Mabsadd L
.5381 .82718 0 -2.5 Mabsadd L
.5381 .82718 -1.875 0 Mabsadd L closepath F
.53905 .82718 -1.875 0 Mabsadd m
.53905 .82718 0 2.5 Mabsadd L
.53905 .82718 1.875 0 Mabsadd L
.53905 .82718 0 -2.5 Mabsadd L
.53905 .82718 -1.875 0 Mabsadd L closepath F
.54 .82718 -1.875 0 Mabsadd m
.54 .82718 0 2.5 Mabsadd L
.54 .82718 1.875 0 Mabsadd L
.54 .82718 0 -2.5 Mabsadd L
.54 .82718 -1.875 0 Mabsadd L closepath F
.54095 .82718 -1.875 0 Mabsadd m
.54095 .82718 0 2.5 Mabsadd L
.54095 .82718 1.875 0 Mabsadd L
.54095 .82718 0 -2.5 Mabsadd L
.54095 .82718 -1.875 0 Mabsadd L closepath F
.5419 .82718 -1.875 0 Mabsadd m
.5419 .82718 0 2.5 Mabsadd L
.5419 .82718 1.875 0 Mabsadd L
.5419 .82718 0 -2.5 Mabsadd L
.5419 .82718 -1.875 0 Mabsadd L closepath F
.54286 .82718 -1.875 0 Mabsadd m
.54286 .82718 0 2.5 Mabsadd L
.54286 .82718 1.875 0 Mabsadd L
.54286 .82718 0 -2.5 Mabsadd L
.54286 .82718 -1.875 0 Mabsadd L closepath F
.54381 .82718 -1.875 0 Mabsadd m
.54381 .82718 0 2.5 Mabsadd L
.54381 .82718 1.875 0 Mabsadd L
.54381 .82718 0 -2.5 Mabsadd L
.54381 .82718 -1.875 0 Mabsadd L closepath F
.54476 .82718 -1.875 0 Mabsadd m
.54476 .82718 0 2.5 Mabsadd L
.54476 .82718 1.875 0 Mabsadd L
.54476 .82718 0 -2.5 Mabsadd L
.54476 .82718 -1.875 0 Mabsadd L closepath F
.54571 .82718 -1.875 0 Mabsadd m
.54571 .82718 0 2.5 Mabsadd L
.54571 .82718 1.875 0 Mabsadd L
.54571 .82718 0 -2.5 Mabsadd L
.54571 .82718 -1.875 0 Mabsadd L closepath F
.54667 .82718 -1.875 0 Mabsadd m
.54667 .82718 0 2.5 Mabsadd L
.54667 .82718 1.875 0 Mabsadd L
.54667 .82718 0 -2.5 Mabsadd L
.54667 .82718 -1.875 0 Mabsadd L closepath F
.54762 .82718 -1.875 0 Mabsadd m
.54762 .82718 0 2.5 Mabsadd L
.54762 .82718 1.875 0 Mabsadd L
.54762 .82718 0 -2.5 Mabsadd L
.54762 .82718 -1.875 0 Mabsadd L closepath F
.54857 .82718 -1.875 0 Mabsadd m
.54857 .82718 0 2.5 Mabsadd L
.54857 .82718 1.875 0 Mabsadd L
.54857 .82718 0 -2.5 Mabsadd L
.54857 .82718 -1.875 0 Mabsadd L closepath F
.54952 .82718 -1.875 0 Mabsadd m
.54952 .82718 0 2.5 Mabsadd L
.54952 .82718 1.875 0 Mabsadd L
.54952 .82718 0 -2.5 Mabsadd L
.54952 .82718 -1.875 0 Mabsadd L closepath F
.55048 .82717 -1.875 0 Mabsadd m
.55048 .82717 0 2.5 Mabsadd L
.55048 .82717 1.875 0 Mabsadd L
.55048 .82717 0 -2.5 Mabsadd L
.55048 .82717 -1.875 0 Mabsadd L closepath F
.55143 .82716 -1.875 0 Mabsadd m
.55143 .82716 0 2.5 Mabsadd L
.55143 .82716 1.875 0 Mabsadd L
.55143 .82716 0 -2.5 Mabsadd L
.55143 .82716 -1.875 0 Mabsadd L closepath F
.55238 .82711 -1.875 0 Mabsadd m
.55238 .82711 0 2.5 Mabsadd L
.55238 .82711 1.875 0 Mabsadd L
.55238 .82711 0 -2.5 Mabsadd L
.55238 .82711 -1.875 0 Mabsadd L closepath F
.55333 .82699 -1.875 0 Mabsadd m
.55333 .82699 0 2.5 Mabsadd L
.55333 .82699 1.875 0 Mabsadd L
.55333 .82699 0 -2.5 Mabsadd L
.55333 .82699 -1.875 0 Mabsadd L closepath F
.55429 .82683 -1.875 0 Mabsadd m
.55429 .82683 0 2.5 Mabsadd L
.55429 .82683 1.875 0 Mabsadd L
.55429 .82683 0 -2.5 Mabsadd L
.55429 .82683 -1.875 0 Mabsadd L closepath F
.55524 .8251 -1.875 0 Mabsadd m
.55524 .8251 0 2.5 Mabsadd L
.55524 .8251 1.875 0 Mabsadd L
.55524 .8251 0 -2.5 Mabsadd L
.55524 .8251 -1.875 0 Mabsadd L closepath F
.55619 .78542 -1.875 0 Mabsadd m
.55619 .78542 0 2.5 Mabsadd L
.55619 .78542 1.875 0 Mabsadd L
.55619 .78542 0 -2.5 Mabsadd L
.55619 .78542 -1.875 0 Mabsadd L closepath F
.55714 .71817 -1.875 0 Mabsadd m
.55714 .71817 0 2.5 Mabsadd L
.55714 .71817 1.875 0 Mabsadd L
.55714 .71817 0 -2.5 Mabsadd L
.55714 .71817 -1.875 0 Mabsadd L closepath F
.5581 .41401 -1.875 0 Mabsadd m
.5581 .41401 0 2.5 Mabsadd L
.5581 .41401 1.875 0 Mabsadd L
.5581 .41401 0 -2.5 Mabsadd L
.5581 .41401 -1.875 0 Mabsadd L closepath F
.55905 .87864 -1.875 0 Mabsadd m
.55905 .87864 0 2.5 Mabsadd L
.55905 .87864 1.875 0 Mabsadd L
.55905 .87864 0 -2.5 Mabsadd L
.55905 .87864 -1.875 0 Mabsadd L closepath F
.56 .81555 -1.875 0 Mabsadd m
.56 .81555 0 2.5 Mabsadd L
.56 .81555 1.875 0 Mabsadd L
.56 .81555 0 -2.5 Mabsadd L
.56 .81555 -1.875 0 Mabsadd L closepath F
.56095 .91423 -1.875 0 Mabsadd m
.56095 .91423 0 2.5 Mabsadd L
.56095 .91423 1.875 0 Mabsadd L
.56095 .91423 0 -2.5 Mabsadd L
.56095 .91423 -1.875 0 Mabsadd L closepath F
.5619 .92749 -1.875 0 Mabsadd m
.5619 .92749 0 2.5 Mabsadd L
.5619 .92749 1.875 0 Mabsadd L
.5619 .92749 0 -2.5 Mabsadd L
.5619 .92749 -1.875 0 Mabsadd L closepath F
.56286 .92788 -1.875 0 Mabsadd m
.56286 .92788 0 2.5 Mabsadd L
.56286 .92788 1.875 0 Mabsadd L
.56286 .92788 0 -2.5 Mabsadd L
.56286 .92788 -1.875 0 Mabsadd L closepath F
.56381 .92803 -1.875 0 Mabsadd m
.56381 .92803 0 2.5 Mabsadd L
.56381 .92803 1.875 0 Mabsadd L
.56381 .92803 0 -2.5 Mabsadd L
.56381 .92803 -1.875 0 Mabsadd L closepath F
.56476 .92812 -1.875 0 Mabsadd m
.56476 .92812 0 2.5 Mabsadd L
.56476 .92812 1.875 0 Mabsadd L
.56476 .92812 0 -2.5 Mabsadd L
.56476 .92812 -1.875 0 Mabsadd L closepath F
.56571 .92814 -1.875 0 Mabsadd m
.56571 .92814 0 2.5 Mabsadd L
.56571 .92814 1.875 0 Mabsadd L
.56571 .92814 0 -2.5 Mabsadd L
.56571 .92814 -1.875 0 Mabsadd L closepath F
.56667 .92814 -1.875 0 Mabsadd m
.56667 .92814 0 2.5 Mabsadd L
.56667 .92814 1.875 0 Mabsadd L
.56667 .92814 0 -2.5 Mabsadd L
.56667 .92814 -1.875 0 Mabsadd L closepath F
.56762 .92815 -1.875 0 Mabsadd m
.56762 .92815 0 2.5 Mabsadd L
.56762 .92815 1.875 0 Mabsadd L
.56762 .92815 0 -2.5 Mabsadd L
.56762 .92815 -1.875 0 Mabsadd L closepath F
.56857 .92816 -1.875 0 Mabsadd m
.56857 .92816 0 2.5 Mabsadd L
.56857 .92816 1.875 0 Mabsadd L
.56857 .92816 0 -2.5 Mabsadd L
.56857 .92816 -1.875 0 Mabsadd L closepath F
.56952 .92816 -1.875 0 Mabsadd m
.56952 .92816 0 2.5 Mabsadd L
.56952 .92816 1.875 0 Mabsadd L
.56952 .92816 0 -2.5 Mabsadd L
.56952 .92816 -1.875 0 Mabsadd L closepath F
.57048 .92816 -1.875 0 Mabsadd m
.57048 .92816 0 2.5 Mabsadd L
.57048 .92816 1.875 0 Mabsadd L
.57048 .92816 0 -2.5 Mabsadd L
.57048 .92816 -1.875 0 Mabsadd L closepath F
.57143 .92816 -1.875 0 Mabsadd m
.57143 .92816 0 2.5 Mabsadd L
.57143 .92816 1.875 0 Mabsadd L
.57143 .92816 0 -2.5 Mabsadd L
.57143 .92816 -1.875 0 Mabsadd L closepath F
.57238 .92816 -1.875 0 Mabsadd m
.57238 .92816 0 2.5 Mabsadd L
.57238 .92816 1.875 0 Mabsadd L
.57238 .92816 0 -2.5 Mabsadd L
.57238 .92816 -1.875 0 Mabsadd L closepath F
.57333 .92816 -1.875 0 Mabsadd m
.57333 .92816 0 2.5 Mabsadd L
.57333 .92816 1.875 0 Mabsadd L
.57333 .92816 0 -2.5 Mabsadd L
.57333 .92816 -1.875 0 Mabsadd L closepath F
.57429 .92816 -1.875 0 Mabsadd m
.57429 .92816 0 2.5 Mabsadd L
.57429 .92816 1.875 0 Mabsadd L
.57429 .92816 0 -2.5 Mabsadd L
.57429 .92816 -1.875 0 Mabsadd L closepath F
.57524 .92816 -1.875 0 Mabsadd m
.57524 .92816 0 2.5 Mabsadd L
.57524 .92816 1.875 0 Mabsadd L
.57524 .92816 0 -2.5 Mabsadd L
.57524 .92816 -1.875 0 Mabsadd L closepath F
.57619 .92816 -1.875 0 Mabsadd m
.57619 .92816 0 2.5 Mabsadd L
.57619 .92816 1.875 0 Mabsadd L
.57619 .92816 0 -2.5 Mabsadd L
.57619 .92816 -1.875 0 Mabsadd L closepath F
.57714 .92816 -1.875 0 Mabsadd m
.57714 .92816 0 2.5 Mabsadd L
.57714 .92816 1.875 0 Mabsadd L
.57714 .92816 0 -2.5 Mabsadd L
.57714 .92816 -1.875 0 Mabsadd L closepath F
.5781 .92816 -1.875 0 Mabsadd m
.5781 .92816 0 2.5 Mabsadd L
.5781 .92816 1.875 0 Mabsadd L
.5781 .92816 0 -2.5 Mabsadd L
.5781 .92816 -1.875 0 Mabsadd L closepath F
.57905 .92816 -1.875 0 Mabsadd m
.57905 .92816 0 2.5 Mabsadd L
.57905 .92816 1.875 0 Mabsadd L
.57905 .92816 0 -2.5 Mabsadd L
.57905 .92816 -1.875 0 Mabsadd L closepath F
.58 .92816 -1.875 0 Mabsadd m
.58 .92816 0 2.5 Mabsadd L
.58 .92816 1.875 0 Mabsadd L
.58 .92816 0 -2.5 Mabsadd L
.58 .92816 -1.875 0 Mabsadd L closepath F
.58095 .92816 -1.875 0 Mabsadd m
.58095 .92816 0 2.5 Mabsadd L
.58095 .92816 1.875 0 Mabsadd L
.58095 .92816 0 -2.5 Mabsadd L
.58095 .92816 -1.875 0 Mabsadd L closepath F
.5819 .92815 -1.875 0 Mabsadd m
.5819 .92815 0 2.5 Mabsadd L
.5819 .92815 1.875 0 Mabsadd L
.5819 .92815 0 -2.5 Mabsadd L
.5819 .92815 -1.875 0 Mabsadd L closepath F
.58286 .92815 -1.875 0 Mabsadd m
.58286 .92815 0 2.5 Mabsadd L
.58286 .92815 1.875 0 Mabsadd L
.58286 .92815 0 -2.5 Mabsadd L
.58286 .92815 -1.875 0 Mabsadd L closepath F
.58381 .92815 -1.875 0 Mabsadd m
.58381 .92815 0 2.5 Mabsadd L
.58381 .92815 1.875 0 Mabsadd L
.58381 .92815 0 -2.5 Mabsadd L
.58381 .92815 -1.875 0 Mabsadd L closepath F
.58476 .92815 -1.875 0 Mabsadd m
.58476 .92815 0 2.5 Mabsadd L
.58476 .92815 1.875 0 Mabsadd L
.58476 .92815 0 -2.5 Mabsadd L
.58476 .92815 -1.875 0 Mabsadd L closepath F
.58571 .92816 -1.875 0 Mabsadd m
.58571 .92816 0 2.5 Mabsadd L
.58571 .92816 1.875 0 Mabsadd L
.58571 .92816 0 -2.5 Mabsadd L
.58571 .92816 -1.875 0 Mabsadd L closepath F
.58667 .92815 -1.875 0 Mabsadd m
.58667 .92815 0 2.5 Mabsadd L
.58667 .92815 1.875 0 Mabsadd L
.58667 .92815 0 -2.5 Mabsadd L
.58667 .92815 -1.875 0 Mabsadd L closepath F
.58762 .92815 -1.875 0 Mabsadd m
.58762 .92815 0 2.5 Mabsadd L
.58762 .92815 1.875 0 Mabsadd L
.58762 .92815 0 -2.5 Mabsadd L
.58762 .92815 -1.875 0 Mabsadd L closepath F
.58857 .92815 -1.875 0 Mabsadd m
.58857 .92815 0 2.5 Mabsadd L
.58857 .92815 1.875 0 Mabsadd L
.58857 .92815 0 -2.5 Mabsadd L
.58857 .92815 -1.875 0 Mabsadd L closepath F
.58952 .92815 -1.875 0 Mabsadd m
.58952 .92815 0 2.5 Mabsadd L
.58952 .92815 1.875 0 Mabsadd L
.58952 .92815 0 -2.5 Mabsadd L
.58952 .92815 -1.875 0 Mabsadd L closepath F
.59048 .92815 -1.875 0 Mabsadd m
.59048 .92815 0 2.5 Mabsadd L
.59048 .92815 1.875 0 Mabsadd L
.59048 .92815 0 -2.5 Mabsadd L
.59048 .92815 -1.875 0 Mabsadd L closepath F
.59143 .92816 -1.875 0 Mabsadd m
.59143 .92816 0 2.5 Mabsadd L
.59143 .92816 1.875 0 Mabsadd L
.59143 .92816 0 -2.5 Mabsadd L
.59143 .92816 -1.875 0 Mabsadd L closepath F
.59238 .92815 -1.875 0 Mabsadd m
.59238 .92815 0 2.5 Mabsadd L
.59238 .92815 1.875 0 Mabsadd L
.59238 .92815 0 -2.5 Mabsadd L
.59238 .92815 -1.875 0 Mabsadd L closepath F
.59333 .92815 -1.875 0 Mabsadd m
.59333 .92815 0 2.5 Mabsadd L
.59333 .92815 1.875 0 Mabsadd L
.59333 .92815 0 -2.5 Mabsadd L
.59333 .92815 -1.875 0 Mabsadd L closepath F
.59429 .92815 -1.875 0 Mabsadd m
.59429 .92815 0 2.5 Mabsadd L
.59429 .92815 1.875 0 Mabsadd L
.59429 .92815 0 -2.5 Mabsadd L
.59429 .92815 -1.875 0 Mabsadd L closepath F
.59524 .92816 -1.875 0 Mabsadd m
.59524 .92816 0 2.5 Mabsadd L
.59524 .92816 1.875 0 Mabsadd L
.59524 .92816 0 -2.5 Mabsadd L
.59524 .92816 -1.875 0 Mabsadd L closepath F
.59619 .92816 -1.875 0 Mabsadd m
.59619 .92816 0 2.5 Mabsadd L
.59619 .92816 1.875 0 Mabsadd L
.59619 .92816 0 -2.5 Mabsadd L
.59619 .92816 -1.875 0 Mabsadd L closepath F
.59714 .92816 -1.875 0 Mabsadd m
.59714 .92816 0 2.5 Mabsadd L
.59714 .92816 1.875 0 Mabsadd L
.59714 .92816 0 -2.5 Mabsadd L
.59714 .92816 -1.875 0 Mabsadd L closepath F
.5981 .92815 -1.875 0 Mabsadd m
.5981 .92815 0 2.5 Mabsadd L
.5981 .92815 1.875 0 Mabsadd L
.5981 .92815 0 -2.5 Mabsadd L
.5981 .92815 -1.875 0 Mabsadd L closepath F
.59905 .92815 -1.875 0 Mabsadd m
.59905 .92815 0 2.5 Mabsadd L
.59905 .92815 1.875 0 Mabsadd L
.59905 .92815 0 -2.5 Mabsadd L
.59905 .92815 -1.875 0 Mabsadd L closepath F
.6 .92815 -1.875 0 Mabsadd m
.6 .92815 0 2.5 Mabsadd L
.6 .92815 1.875 0 Mabsadd L
.6 .92815 0 -2.5 Mabsadd L
.6 .92815 -1.875 0 Mabsadd L closepath F
.60095 .92815 -1.875 0 Mabsadd m
.60095 .92815 0 2.5 Mabsadd L
.60095 .92815 1.875 0 Mabsadd L
.60095 .92815 0 -2.5 Mabsadd L
.60095 .92815 -1.875 0 Mabsadd L closepath F
.6019 .92815 -1.875 0 Mabsadd m
.6019 .92815 0 2.5 Mabsadd L
.6019 .92815 1.875 0 Mabsadd L
.6019 .92815 0 -2.5 Mabsadd L
.6019 .92815 -1.875 0 Mabsadd L closepath F
.60286 .92815 -1.875 0 Mabsadd m
.60286 .92815 0 2.5 Mabsadd L
.60286 .92815 1.875 0 Mabsadd L
.60286 .92815 0 -2.5 Mabsadd L
.60286 .92815 -1.875 0 Mabsadd L closepath F
.60381 .92815 -1.875 0 Mabsadd m
.60381 .92815 0 2.5 Mabsadd L
.60381 .92815 1.875 0 Mabsadd L
.60381 .92815 0 -2.5 Mabsadd L
.60381 .92815 -1.875 0 Mabsadd L closepath F
.60476 .92814 -1.875 0 Mabsadd m
.60476 .92814 0 2.5 Mabsadd L
.60476 .92814 1.875 0 Mabsadd L
.60476 .92814 0 -2.5 Mabsadd L
.60476 .92814 -1.875 0 Mabsadd L closepath F
.60571 .9281 -1.875 0 Mabsadd m
.60571 .9281 0 2.5 Mabsadd L
.60571 .9281 1.875 0 Mabsadd L
.60571 .9281 0 -2.5 Mabsadd L
.60571 .9281 -1.875 0 Mabsadd L closepath F
.60667 .928 -1.875 0 Mabsadd m
.60667 .928 0 2.5 Mabsadd L
.60667 .928 1.875 0 Mabsadd L
.60667 .928 0 -2.5 Mabsadd L
.60667 .928 -1.875 0 Mabsadd L closepath F
.60762 .92783 -1.875 0 Mabsadd m
.60762 .92783 0 2.5 Mabsadd L
.60762 .92783 1.875 0 Mabsadd L
.60762 .92783 0 -2.5 Mabsadd L
.60762 .92783 -1.875 0 Mabsadd L closepath F
.60857 .92677 -1.875 0 Mabsadd m
.60857 .92677 0 2.5 Mabsadd L
.60857 .92677 1.875 0 Mabsadd L
.60857 .92677 0 -2.5 Mabsadd L
.60857 .92677 -1.875 0 Mabsadd L closepath F
.60952 .89777 -1.875 0 Mabsadd m
.60952 .89777 0 2.5 Mabsadd L
.60952 .89777 1.875 0 Mabsadd L
.60952 .89777 0 -2.5 Mabsadd L
.60952 .89777 -1.875 0 Mabsadd L closepath F
.61048 .80672 -1.875 0 Mabsadd m
.61048 .80672 0 2.5 Mabsadd L
.61048 .80672 1.875 0 Mabsadd L
.61048 .80672 0 -2.5 Mabsadd L
.61048 .80672 -1.875 0 Mabsadd L closepath F
.61143 .67403 -1.875 0 Mabsadd m
.61143 .67403 0 2.5 Mabsadd L
.61143 .67403 1.875 0 Mabsadd L
.61143 .67403 0 -2.5 Mabsadd L
.61143 .67403 -1.875 0 Mabsadd L closepath F
.61238 .8619 -1.875 0 Mabsadd m
.61238 .8619 0 2.5 Mabsadd L
.61238 .8619 1.875 0 Mabsadd L
.61238 .8619 0 -2.5 Mabsadd L
.61238 .8619 -1.875 0 Mabsadd L closepath F
.61333 .85437 -1.875 0 Mabsadd m
.61333 .85437 0 2.5 Mabsadd L
.61333 .85437 1.875 0 Mabsadd L
.61333 .85437 0 -2.5 Mabsadd L
.61333 .85437 -1.875 0 Mabsadd L closepath F
.61429 .95594 -1.875 0 Mabsadd m
.61429 .95594 0 2.5 Mabsadd L
.61429 .95594 1.875 0 Mabsadd L
.61429 .95594 0 -2.5 Mabsadd L
.61429 .95594 -1.875 0 Mabsadd L closepath F
.61524 .97529 -1.875 0 Mabsadd m
.61524 .97529 0 2.5 Mabsadd L
.61524 .97529 1.875 0 Mabsadd L
.61524 .97529 0 -2.5 Mabsadd L
.61524 .97529 -1.875 0 Mabsadd L closepath F
.61619 .97589 -1.875 0 Mabsadd m
.61619 .97589 0 2.5 Mabsadd L
.61619 .97589 1.875 0 Mabsadd L
.61619 .97589 0 -2.5 Mabsadd L
.61619 .97589 -1.875 0 Mabsadd L closepath F
.61714 .97605 -1.875 0 Mabsadd m
.61714 .97605 0 2.5 Mabsadd L
.61714 .97605 1.875 0 Mabsadd L
.61714 .97605 0 -2.5 Mabsadd L
.61714 .97605 -1.875 0 Mabsadd L closepath F
.6181 .97614 -1.875 0 Mabsadd m
.6181 .97614 0 2.5 Mabsadd L
.6181 .97614 1.875 0 Mabsadd L
.6181 .97614 0 -2.5 Mabsadd L
.6181 .97614 -1.875 0 Mabsadd L closepath F
.61905 .97617 -1.875 0 Mabsadd m
.61905 .97617 0 2.5 Mabsadd L
.61905 .97617 1.875 0 Mabsadd L
.61905 .97617 0 -2.5 Mabsadd L
.61905 .97617 -1.875 0 Mabsadd L closepath F
.62 .97618 -1.875 0 Mabsadd m
.62 .97618 0 2.5 Mabsadd L
.62 .97618 1.875 0 Mabsadd L
.62 .97618 0 -2.5 Mabsadd L
.62 .97618 -1.875 0 Mabsadd L closepath F
.62095 .97619 -1.875 0 Mabsadd m
.62095 .97619 0 2.5 Mabsadd L
.62095 .97619 1.875 0 Mabsadd L
.62095 .97619 0 -2.5 Mabsadd L
.62095 .97619 -1.875 0 Mabsadd L closepath F
.6219 .97619 -1.875 0 Mabsadd m
.6219 .97619 0 2.5 Mabsadd L
.6219 .97619 1.875 0 Mabsadd L
.6219 .97619 0 -2.5 Mabsadd L
.6219 .97619 -1.875 0 Mabsadd L closepath F
.62286 .97619 -1.875 0 Mabsadd m
.62286 .97619 0 2.5 Mabsadd L
.62286 .97619 1.875 0 Mabsadd L
.62286 .97619 0 -2.5 Mabsadd L
.62286 .97619 -1.875 0 Mabsadd L closepath F
.62381 .97619 -1.875 0 Mabsadd m
.62381 .97619 0 2.5 Mabsadd L
.62381 .97619 1.875 0 Mabsadd L
.62381 .97619 0 -2.5 Mabsadd L
.62381 .97619 -1.875 0 Mabsadd L closepath F
.62476 .97619 -1.875 0 Mabsadd m
.62476 .97619 0 2.5 Mabsadd L
.62476 .97619 1.875 0 Mabsadd L
.62476 .97619 0 -2.5 Mabsadd L
.62476 .97619 -1.875 0 Mabsadd L closepath F
.62571 .97619 -1.875 0 Mabsadd m
.62571 .97619 0 2.5 Mabsadd L
.62571 .97619 1.875 0 Mabsadd L
.62571 .97619 0 -2.5 Mabsadd L
.62571 .97619 -1.875 0 Mabsadd L closepath F
.62667 .97619 -1.875 0 Mabsadd m
.62667 .97619 0 2.5 Mabsadd L
.62667 .97619 1.875 0 Mabsadd L
.62667 .97619 0 -2.5 Mabsadd L
.62667 .97619 -1.875 0 Mabsadd L closepath F
.62762 .97619 -1.875 0 Mabsadd m
.62762 .97619 0 2.5 Mabsadd L
.62762 .97619 1.875 0 Mabsadd L
.62762 .97619 0 -2.5 Mabsadd L
.62762 .97619 -1.875 0 Mabsadd L closepath F
.62857 .97619 -1.875 0 Mabsadd m
.62857 .97619 0 2.5 Mabsadd L
.62857 .97619 1.875 0 Mabsadd L
.62857 .97619 0 -2.5 Mabsadd L
.62857 .97619 -1.875 0 Mabsadd L closepath F
.62952 .97619 -1.875 0 Mabsadd m
.62952 .97619 0 2.5 Mabsadd L
.62952 .97619 1.875 0 Mabsadd L
.62952 .97619 0 -2.5 Mabsadd L
.62952 .97619 -1.875 0 Mabsadd L closepath F
.63048 .97619 -1.875 0 Mabsadd m
.63048 .97619 0 2.5 Mabsadd L
.63048 .97619 1.875 0 Mabsadd L
.63048 .97619 0 -2.5 Mabsadd L
.63048 .97619 -1.875 0 Mabsadd L closepath F
.63143 .97619 -1.875 0 Mabsadd m
.63143 .97619 0 2.5 Mabsadd L
.63143 .97619 1.875 0 Mabsadd L
.63143 .97619 0 -2.5 Mabsadd L
.63143 .97619 -1.875 0 Mabsadd L closepath F
.63238 .97619 -1.875 0 Mabsadd m
.63238 .97619 0 2.5 Mabsadd L
.63238 .97619 1.875 0 Mabsadd L
.63238 .97619 0 -2.5 Mabsadd L
.63238 .97619 -1.875 0 Mabsadd L closepath F
.63333 .97619 -1.875 0 Mabsadd m
.63333 .97619 0 2.5 Mabsadd L
.63333 .97619 1.875 0 Mabsadd L
.63333 .97619 0 -2.5 Mabsadd L
.63333 .97619 -1.875 0 Mabsadd L closepath F
.63429 .97619 -1.875 0 Mabsadd m
.63429 .97619 0 2.5 Mabsadd L
.63429 .97619 1.875 0 Mabsadd L
.63429 .97619 0 -2.5 Mabsadd L
.63429 .97619 -1.875 0 Mabsadd L closepath F
.63524 .97619 -1.875 0 Mabsadd m
.63524 .97619 0 2.5 Mabsadd L
.63524 .97619 1.875 0 Mabsadd L
.63524 .97619 0 -2.5 Mabsadd L
.63524 .97619 -1.875 0 Mabsadd L closepath F
.63619 .97619 -1.875 0 Mabsadd m
.63619 .97619 0 2.5 Mabsadd L
.63619 .97619 1.875 0 Mabsadd L
.63619 .97619 0 -2.5 Mabsadd L
.63619 .97619 -1.875 0 Mabsadd L closepath F
.63714 .97619 -1.875 0 Mabsadd m
.63714 .97619 0 2.5 Mabsadd L
.63714 .97619 1.875 0 Mabsadd L
.63714 .97619 0 -2.5 Mabsadd L
.63714 .97619 -1.875 0 Mabsadd L closepath F
.6381 .97619 -1.875 0 Mabsadd m
.6381 .97619 0 2.5 Mabsadd L
.6381 .97619 1.875 0 Mabsadd L
.6381 .97619 0 -2.5 Mabsadd L
.6381 .97619 -1.875 0 Mabsadd L closepath F
.63905 .97619 -1.875 0 Mabsadd m
.63905 .97619 0 2.5 Mabsadd L
.63905 .97619 1.875 0 Mabsadd L
.63905 .97619 0 -2.5 Mabsadd L
.63905 .97619 -1.875 0 Mabsadd L closepath F
.64 .97619 -1.875 0 Mabsadd m
.64 .97619 0 2.5 Mabsadd L
.64 .97619 1.875 0 Mabsadd L
.64 .97619 0 -2.5 Mabsadd L
.64 .97619 -1.875 0 Mabsadd L closepath F
.64095 .97619 -1.875 0 Mabsadd m
.64095 .97619 0 2.5 Mabsadd L
.64095 .97619 1.875 0 Mabsadd L
.64095 .97619 0 -2.5 Mabsadd L
.64095 .97619 -1.875 0 Mabsadd L closepath F
.6419 .97619 -1.875 0 Mabsadd m
.6419 .97619 0 2.5 Mabsadd L
.6419 .97619 1.875 0 Mabsadd L
.6419 .97619 0 -2.5 Mabsadd L
.6419 .97619 -1.875 0 Mabsadd L closepath F
.64286 .97619 -1.875 0 Mabsadd m
.64286 .97619 0 2.5 Mabsadd L
.64286 .97619 1.875 0 Mabsadd L
.64286 .97619 0 -2.5 Mabsadd L
.64286 .97619 -1.875 0 Mabsadd L closepath F
.64381 .97619 -1.875 0 Mabsadd m
.64381 .97619 0 2.5 Mabsadd L
.64381 .97619 1.875 0 Mabsadd L
.64381 .97619 0 -2.5 Mabsadd L
.64381 .97619 -1.875 0 Mabsadd L closepath F
.64476 .97619 -1.875 0 Mabsadd m
.64476 .97619 0 2.5 Mabsadd L
.64476 .97619 1.875 0 Mabsadd L
.64476 .97619 0 -2.5 Mabsadd L
.64476 .97619 -1.875 0 Mabsadd L closepath F
.64571 .97619 -1.875 0 Mabsadd m
.64571 .97619 0 2.5 Mabsadd L
.64571 .97619 1.875 0 Mabsadd L
.64571 .97619 0 -2.5 Mabsadd L
.64571 .97619 -1.875 0 Mabsadd L closepath F
.64667 .97619 -1.875 0 Mabsadd m
.64667 .97619 0 2.5 Mabsadd L
.64667 .97619 1.875 0 Mabsadd L
.64667 .97619 0 -2.5 Mabsadd L
.64667 .97619 -1.875 0 Mabsadd L closepath F
.64762 .97619 -1.875 0 Mabsadd m
.64762 .97619 0 2.5 Mabsadd L
.64762 .97619 1.875 0 Mabsadd L
.64762 .97619 0 -2.5 Mabsadd L
.64762 .97619 -1.875 0 Mabsadd L closepath F
.64857 .97619 -1.875 0 Mabsadd m
.64857 .97619 0 2.5 Mabsadd L
.64857 .97619 1.875 0 Mabsadd L
.64857 .97619 0 -2.5 Mabsadd L
.64857 .97619 -1.875 0 Mabsadd L closepath F
.64952 .97619 -1.875 0 Mabsadd m
.64952 .97619 0 2.5 Mabsadd L
.64952 .97619 1.875 0 Mabsadd L
.64952 .97619 0 -2.5 Mabsadd L
.64952 .97619 -1.875 0 Mabsadd L closepath F
.65048 .97619 -1.875 0 Mabsadd m
.65048 .97619 0 2.5 Mabsadd L
.65048 .97619 1.875 0 Mabsadd L
.65048 .97619 0 -2.5 Mabsadd L
.65048 .97619 -1.875 0 Mabsadd L closepath F
.65143 .97619 -1.875 0 Mabsadd m
.65143 .97619 0 2.5 Mabsadd L
.65143 .97619 1.875 0 Mabsadd L
.65143 .97619 0 -2.5 Mabsadd L
.65143 .97619 -1.875 0 Mabsadd L closepath F
.65238 .97619 -1.875 0 Mabsadd m
.65238 .97619 0 2.5 Mabsadd L
.65238 .97619 1.875 0 Mabsadd L
.65238 .97619 0 -2.5 Mabsadd L
.65238 .97619 -1.875 0 Mabsadd L closepath F
.65333 .97619 -1.875 0 Mabsadd m
.65333 .97619 0 2.5 Mabsadd L
.65333 .97619 1.875 0 Mabsadd L
.65333 .97619 0 -2.5 Mabsadd L
.65333 .97619 -1.875 0 Mabsadd L closepath F
.65429 .97619 -1.875 0 Mabsadd m
.65429 .97619 0 2.5 Mabsadd L
.65429 .97619 1.875 0 Mabsadd L
.65429 .97619 0 -2.5 Mabsadd L
.65429 .97619 -1.875 0 Mabsadd L closepath F
.65524 .97619 -1.875 0 Mabsadd m
.65524 .97619 0 2.5 Mabsadd L
.65524 .97619 1.875 0 Mabsadd L
.65524 .97619 0 -2.5 Mabsadd L
.65524 .97619 -1.875 0 Mabsadd L closepath F
.65619 .97619 -1.875 0 Mabsadd m
.65619 .97619 0 2.5 Mabsadd L
.65619 .97619 1.875 0 Mabsadd L
.65619 .97619 0 -2.5 Mabsadd L
.65619 .97619 -1.875 0 Mabsadd L closepath F
.65714 .97618 -1.875 0 Mabsadd m
.65714 .97618 0 2.5 Mabsadd L
.65714 .97618 1.875 0 Mabsadd L
.65714 .97618 0 -2.5 Mabsadd L
.65714 .97618 -1.875 0 Mabsadd L closepath F
.6581 .97617 -1.875 0 Mabsadd m
.6581 .97617 0 2.5 Mabsadd L
.6581 .97617 1.875 0 Mabsadd L
.6581 .97617 0 -2.5 Mabsadd L
.6581 .97617 -1.875 0 Mabsadd L closepath F
.65905 .97614 -1.875 0 Mabsadd m
.65905 .97614 0 2.5 Mabsadd L
.65905 .97614 1.875 0 Mabsadd L
.65905 .97614 0 -2.5 Mabsadd L
.65905 .97614 -1.875 0 Mabsadd L closepath F
.66 .97604 -1.875 0 Mabsadd m
.66 .97604 0 2.5 Mabsadd L
.66 .97604 1.875 0 Mabsadd L
.66 .97604 0 -2.5 Mabsadd L
.66 .97604 -1.875 0 Mabsadd L closepath F
.66095 .97588 -1.875 0 Mabsadd m
.66095 .97588 0 2.5 Mabsadd L
.66095 .97588 1.875 0 Mabsadd L
.66095 .97588 0 -2.5 Mabsadd L
.66095 .97588 -1.875 0 Mabsadd L closepath F
.6619 .97524 -1.875 0 Mabsadd m
.6619 .97524 0 2.5 Mabsadd L
.6619 .97524 1.875 0 Mabsadd L
.6619 .97524 0 -2.5 Mabsadd L
.6619 .97524 -1.875 0 Mabsadd L closepath F
.66286 .95468 -1.875 0 Mabsadd m
.66286 .95468 0 2.5 Mabsadd L
.66286 .95468 1.875 0 Mabsadd L
.66286 .95468 0 -2.5 Mabsadd L
.66286 .95468 -1.875 0 Mabsadd L closepath F
.66381 .85359 -1.875 0 Mabsadd m
.66381 .85359 0 2.5 Mabsadd L
.66381 .85359 1.875 0 Mabsadd L
.66381 .85359 0 -2.5 Mabsadd L
.66381 .85359 -1.875 0 Mabsadd L closepath F
.66476 .84607 -1.875 0 Mabsadd m
.66476 .84607 0 2.5 Mabsadd L
.66476 .84607 1.875 0 Mabsadd L
.66476 .84607 0 -2.5 Mabsadd L
.66476 .84607 -1.875 0 Mabsadd L closepath F
.66571 .71078 -1.875 0 Mabsadd m
.66571 .71078 0 2.5 Mabsadd L
.66571 .71078 1.875 0 Mabsadd L
.66571 .71078 0 -2.5 Mabsadd L
.66571 .71078 -1.875 0 Mabsadd L closepath F
.66667 .81881 -1.875 0 Mabsadd m
.66667 .81881 0 2.5 Mabsadd L
.66667 .81881 1.875 0 Mabsadd L
.66667 .81881 0 -2.5 Mabsadd L
.66667 .81881 -1.875 0 Mabsadd L closepath F
.66762 .91245 -1.875 0 Mabsadd m
.66762 .91245 0 2.5 Mabsadd L
.66762 .91245 1.875 0 Mabsadd L
.66762 .91245 0 -2.5 Mabsadd L
.66762 .91245 -1.875 0 Mabsadd L closepath F
.66857 .9399 -1.875 0 Mabsadd m
.66857 .9399 0 2.5 Mabsadd L
.66857 .9399 1.875 0 Mabsadd L
.66857 .9399 0 -2.5 Mabsadd L
.66857 .9399 -1.875 0 Mabsadd L closepath F
.66952 .94086 -1.875 0 Mabsadd m
.66952 .94086 0 2.5 Mabsadd L
.66952 .94086 1.875 0 Mabsadd L
.66952 .94086 0 -2.5 Mabsadd L
.66952 .94086 -1.875 0 Mabsadd L closepath F
.67048 .94103 -1.875 0 Mabsadd m
.67048 .94103 0 2.5 Mabsadd L
.67048 .94103 1.875 0 Mabsadd L
.67048 .94103 0 -2.5 Mabsadd L
.67048 .94103 -1.875 0 Mabsadd L closepath F
.67143 .94113 -1.875 0 Mabsadd m
.67143 .94113 0 2.5 Mabsadd L
.67143 .94113 1.875 0 Mabsadd L
.67143 .94113 0 -2.5 Mabsadd L
.67143 .94113 -1.875 0 Mabsadd L closepath F
.67238 .94117 -1.875 0 Mabsadd m
.67238 .94117 0 2.5 Mabsadd L
.67238 .94117 1.875 0 Mabsadd L
.67238 .94117 0 -2.5 Mabsadd L
.67238 .94117 -1.875 0 Mabsadd L closepath F
.67333 .94118 -1.875 0 Mabsadd m
.67333 .94118 0 2.5 Mabsadd L
.67333 .94118 1.875 0 Mabsadd L
.67333 .94118 0 -2.5 Mabsadd L
.67333 .94118 -1.875 0 Mabsadd L closepath F
.67429 .94118 -1.875 0 Mabsadd m
.67429 .94118 0 2.5 Mabsadd L
.67429 .94118 1.875 0 Mabsadd L
.67429 .94118 0 -2.5 Mabsadd L
.67429 .94118 -1.875 0 Mabsadd L closepath F
.67524 .94119 -1.875 0 Mabsadd m
.67524 .94119 0 2.5 Mabsadd L
.67524 .94119 1.875 0 Mabsadd L
.67524 .94119 0 -2.5 Mabsadd L
.67524 .94119 -1.875 0 Mabsadd L closepath F
.67619 .94119 -1.875 0 Mabsadd m
.67619 .94119 0 2.5 Mabsadd L
.67619 .94119 1.875 0 Mabsadd L
.67619 .94119 0 -2.5 Mabsadd L
.67619 .94119 -1.875 0 Mabsadd L closepath F
.67714 .94119 -1.875 0 Mabsadd m
.67714 .94119 0 2.5 Mabsadd L
.67714 .94119 1.875 0 Mabsadd L
.67714 .94119 0 -2.5 Mabsadd L
.67714 .94119 -1.875 0 Mabsadd L closepath F
.6781 .94119 -1.875 0 Mabsadd m
.6781 .94119 0 2.5 Mabsadd L
.6781 .94119 1.875 0 Mabsadd L
.6781 .94119 0 -2.5 Mabsadd L
.6781 .94119 -1.875 0 Mabsadd L closepath F
.67905 .94119 -1.875 0 Mabsadd m
.67905 .94119 0 2.5 Mabsadd L
.67905 .94119 1.875 0 Mabsadd L
.67905 .94119 0 -2.5 Mabsadd L
.67905 .94119 -1.875 0 Mabsadd L closepath F
.68 .94119 -1.875 0 Mabsadd m
.68 .94119 0 2.5 Mabsadd L
.68 .94119 1.875 0 Mabsadd L
.68 .94119 0 -2.5 Mabsadd L
.68 .94119 -1.875 0 Mabsadd L closepath F
.68095 .94119 -1.875 0 Mabsadd m
.68095 .94119 0 2.5 Mabsadd L
.68095 .94119 1.875 0 Mabsadd L
.68095 .94119 0 -2.5 Mabsadd L
.68095 .94119 -1.875 0 Mabsadd L closepath F
.6819 .94119 -1.875 0 Mabsadd m
.6819 .94119 0 2.5 Mabsadd L
.6819 .94119 1.875 0 Mabsadd L
.6819 .94119 0 -2.5 Mabsadd L
.6819 .94119 -1.875 0 Mabsadd L closepath F
.68286 .94119 -1.875 0 Mabsadd m
.68286 .94119 0 2.5 Mabsadd L
.68286 .94119 1.875 0 Mabsadd L
.68286 .94119 0 -2.5 Mabsadd L
.68286 .94119 -1.875 0 Mabsadd L closepath F
.68381 .94119 -1.875 0 Mabsadd m
.68381 .94119 0 2.5 Mabsadd L
.68381 .94119 1.875 0 Mabsadd L
.68381 .94119 0 -2.5 Mabsadd L
.68381 .94119 -1.875 0 Mabsadd L closepath F
.68476 .94119 -1.875 0 Mabsadd m
.68476 .94119 0 2.5 Mabsadd L
.68476 .94119 1.875 0 Mabsadd L
.68476 .94119 0 -2.5 Mabsadd L
.68476 .94119 -1.875 0 Mabsadd L closepath F
.68571 .94119 -1.875 0 Mabsadd m
.68571 .94119 0 2.5 Mabsadd L
.68571 .94119 1.875 0 Mabsadd L
.68571 .94119 0 -2.5 Mabsadd L
.68571 .94119 -1.875 0 Mabsadd L closepath F
.68667 .94119 -1.875 0 Mabsadd m
.68667 .94119 0 2.5 Mabsadd L
.68667 .94119 1.875 0 Mabsadd L
.68667 .94119 0 -2.5 Mabsadd L
.68667 .94119 -1.875 0 Mabsadd L closepath F
.68762 .94119 -1.875 0 Mabsadd m
.68762 .94119 0 2.5 Mabsadd L
.68762 .94119 1.875 0 Mabsadd L
.68762 .94119 0 -2.5 Mabsadd L
.68762 .94119 -1.875 0 Mabsadd L closepath F
.68857 .94119 -1.875 0 Mabsadd m
.68857 .94119 0 2.5 Mabsadd L
.68857 .94119 1.875 0 Mabsadd L
.68857 .94119 0 -2.5 Mabsadd L
.68857 .94119 -1.875 0 Mabsadd L closepath F
.68952 .94119 -1.875 0 Mabsadd m
.68952 .94119 0 2.5 Mabsadd L
.68952 .94119 1.875 0 Mabsadd L
.68952 .94119 0 -2.5 Mabsadd L
.68952 .94119 -1.875 0 Mabsadd L closepath F
.69048 .94119 -1.875 0 Mabsadd m
.69048 .94119 0 2.5 Mabsadd L
.69048 .94119 1.875 0 Mabsadd L
.69048 .94119 0 -2.5 Mabsadd L
.69048 .94119 -1.875 0 Mabsadd L closepath F
.69143 .94119 -1.875 0 Mabsadd m
.69143 .94119 0 2.5 Mabsadd L
.69143 .94119 1.875 0 Mabsadd L
.69143 .94119 0 -2.5 Mabsadd L
.69143 .94119 -1.875 0 Mabsadd L closepath F
.69238 .94119 -1.875 0 Mabsadd m
.69238 .94119 0 2.5 Mabsadd L
.69238 .94119 1.875 0 Mabsadd L
.69238 .94119 0 -2.5 Mabsadd L
.69238 .94119 -1.875 0 Mabsadd L closepath F
.69333 .94119 -1.875 0 Mabsadd m
.69333 .94119 0 2.5 Mabsadd L
.69333 .94119 1.875 0 Mabsadd L
.69333 .94119 0 -2.5 Mabsadd L
.69333 .94119 -1.875 0 Mabsadd L closepath F
.69429 .94119 -1.875 0 Mabsadd m
.69429 .94119 0 2.5 Mabsadd L
.69429 .94119 1.875 0 Mabsadd L
.69429 .94119 0 -2.5 Mabsadd L
.69429 .94119 -1.875 0 Mabsadd L closepath F
.69524 .94119 -1.875 0 Mabsadd m
.69524 .94119 0 2.5 Mabsadd L
.69524 .94119 1.875 0 Mabsadd L
.69524 .94119 0 -2.5 Mabsadd L
.69524 .94119 -1.875 0 Mabsadd L closepath F
.69619 .94119 -1.875 0 Mabsadd m
.69619 .94119 0 2.5 Mabsadd L
.69619 .94119 1.875 0 Mabsadd L
.69619 .94119 0 -2.5 Mabsadd L
.69619 .94119 -1.875 0 Mabsadd L closepath F
.69714 .94119 -1.875 0 Mabsadd m
.69714 .94119 0 2.5 Mabsadd L
.69714 .94119 1.875 0 Mabsadd L
.69714 .94119 0 -2.5 Mabsadd L
.69714 .94119 -1.875 0 Mabsadd L closepath F
.6981 .94119 -1.875 0 Mabsadd m
.6981 .94119 0 2.5 Mabsadd L
.6981 .94119 1.875 0 Mabsadd L
.6981 .94119 0 -2.5 Mabsadd L
.6981 .94119 -1.875 0 Mabsadd L closepath F
.69905 .94119 -1.875 0 Mabsadd m
.69905 .94119 0 2.5 Mabsadd L
.69905 .94119 1.875 0 Mabsadd L
.69905 .94119 0 -2.5 Mabsadd L
.69905 .94119 -1.875 0 Mabsadd L closepath F
.7 .94119 -1.875 0 Mabsadd m
.7 .94119 0 2.5 Mabsadd L
.7 .94119 1.875 0 Mabsadd L
.7 .94119 0 -2.5 Mabsadd L
.7 .94119 -1.875 0 Mabsadd L closepath F
.70095 .94119 -1.875 0 Mabsadd m
.70095 .94119 0 2.5 Mabsadd L
.70095 .94119 1.875 0 Mabsadd L
.70095 .94119 0 -2.5 Mabsadd L
.70095 .94119 -1.875 0 Mabsadd L closepath F
.7019 .94119 -1.875 0 Mabsadd m
.7019 .94119 0 2.5 Mabsadd L
.7019 .94119 1.875 0 Mabsadd L
.7019 .94119 0 -2.5 Mabsadd L
.7019 .94119 -1.875 0 Mabsadd L closepath F
.70286 .94119 -1.875 0 Mabsadd m
.70286 .94119 0 2.5 Mabsadd L
.70286 .94119 1.875 0 Mabsadd L
.70286 .94119 0 -2.5 Mabsadd L
.70286 .94119 -1.875 0 Mabsadd L closepath F
.70381 .94119 -1.875 0 Mabsadd m
.70381 .94119 0 2.5 Mabsadd L
.70381 .94119 1.875 0 Mabsadd L
.70381 .94119 0 -2.5 Mabsadd L
.70381 .94119 -1.875 0 Mabsadd L closepath F
.70476 .94119 -1.875 0 Mabsadd m
.70476 .94119 0 2.5 Mabsadd L
.70476 .94119 1.875 0 Mabsadd L
.70476 .94119 0 -2.5 Mabsadd L
.70476 .94119 -1.875 0 Mabsadd L closepath F
.70571 .94119 -1.875 0 Mabsadd m
.70571 .94119 0 2.5 Mabsadd L
.70571 .94119 1.875 0 Mabsadd L
.70571 .94119 0 -2.5 Mabsadd L
.70571 .94119 -1.875 0 Mabsadd L closepath F
.70667 .94119 -1.875 0 Mabsadd m
.70667 .94119 0 2.5 Mabsadd L
.70667 .94119 1.875 0 Mabsadd L
.70667 .94119 0 -2.5 Mabsadd L
.70667 .94119 -1.875 0 Mabsadd L closepath F
.70762 .94119 -1.875 0 Mabsadd m
.70762 .94119 0 2.5 Mabsadd L
.70762 .94119 1.875 0 Mabsadd L
.70762 .94119 0 -2.5 Mabsadd L
.70762 .94119 -1.875 0 Mabsadd L closepath F
.70857 .94119 -1.875 0 Mabsadd m
.70857 .94119 0 2.5 Mabsadd L
.70857 .94119 1.875 0 Mabsadd L
.70857 .94119 0 -2.5 Mabsadd L
.70857 .94119 -1.875 0 Mabsadd L closepath F
.70952 .94119 -1.875 0 Mabsadd m
.70952 .94119 0 2.5 Mabsadd L
.70952 .94119 1.875 0 Mabsadd L
.70952 .94119 0 -2.5 Mabsadd L
.70952 .94119 -1.875 0 Mabsadd L closepath F
.71048 .94118 -1.875 0 Mabsadd m
.71048 .94118 0 2.5 Mabsadd L
.71048 .94118 1.875 0 Mabsadd L
.71048 .94118 0 -2.5 Mabsadd L
.71048 .94118 -1.875 0 Mabsadd L closepath F
.71143 .94117 -1.875 0 Mabsadd m
.71143 .94117 0 2.5 Mabsadd L
.71143 .94117 1.875 0 Mabsadd L
.71143 .94117 0 -2.5 Mabsadd L
.71143 .94117 -1.875 0 Mabsadd L closepath F
.71238 .94114 -1.875 0 Mabsadd m
.71238 .94114 0 2.5 Mabsadd L
.71238 .94114 1.875 0 Mabsadd L
.71238 .94114 0 -2.5 Mabsadd L
.71238 .94114 -1.875 0 Mabsadd L closepath F
.71333 .94106 -1.875 0 Mabsadd m
.71333 .94106 0 2.5 Mabsadd L
.71333 .94106 1.875 0 Mabsadd L
.71333 .94106 0 -2.5 Mabsadd L
.71333 .94106 -1.875 0 Mabsadd L closepath F
.71429 .9409 -1.875 0 Mabsadd m
.71429 .9409 0 2.5 Mabsadd L
.71429 .9409 1.875 0 Mabsadd L
.71429 .9409 0 -2.5 Mabsadd L
.71429 .9409 -1.875 0 Mabsadd L closepath F
.71524 .94049 -1.875 0 Mabsadd m
.71524 .94049 0 2.5 Mabsadd L
.71524 .94049 1.875 0 Mabsadd L
.71524 .94049 0 -2.5 Mabsadd L
.71524 .94049 -1.875 0 Mabsadd L closepath F
.71619 .92634 -1.875 0 Mabsadd m
.71619 .92634 0 2.5 Mabsadd L
.71619 .92634 1.875 0 Mabsadd L
.71619 .92634 0 -2.5 Mabsadd L
.71619 .92634 -1.875 0 Mabsadd L closepath F
.71714 .82658 -1.875 0 Mabsadd m
.71714 .82658 0 2.5 Mabsadd L
.71714 .82658 1.875 0 Mabsadd L
.71714 .82658 0 -2.5 Mabsadd L
.71714 .82658 -1.875 0 Mabsadd L closepath F
.7181 .8844 -1.875 0 Mabsadd m
.7181 .8844 0 2.5 Mabsadd L
.7181 .8844 1.875 0 Mabsadd L
.7181 .8844 0 -2.5 Mabsadd L
.7181 .8844 -1.875 0 Mabsadd L closepath F
.71905 .45947 -1.875 0 Mabsadd m
.71905 .45947 0 2.5 Mabsadd L
.71905 .45947 1.875 0 Mabsadd L
.71905 .45947 0 -2.5 Mabsadd L
.71905 .45947 -1.875 0 Mabsadd L closepath F
.72 .7332 -1.875 0 Mabsadd m
.72 .7332 0 2.5 Mabsadd L
.72 .7332 1.875 0 Mabsadd L
.72 .7332 0 -2.5 Mabsadd L
.72 .7332 -1.875 0 Mabsadd L closepath F
.72095 .80537 -1.875 0 Mabsadd m
.72095 .80537 0 2.5 Mabsadd L
.72095 .80537 1.875 0 Mabsadd L
.72095 .80537 0 -2.5 Mabsadd L
.72095 .80537 -1.875 0 Mabsadd L closepath F
.7219 .84313 -1.875 0 Mabsadd m
.7219 .84313 0 2.5 Mabsadd L
.7219 .84313 1.875 0 Mabsadd L
.7219 .84313 0 -2.5 Mabsadd L
.7219 .84313 -1.875 0 Mabsadd L closepath F
.72286 .84472 -1.875 0 Mabsadd m
.72286 .84472 0 2.5 Mabsadd L
.72286 .84472 1.875 0 Mabsadd L
.72286 .84472 0 -2.5 Mabsadd L
.72286 .84472 -1.875 0 Mabsadd L closepath F
.72381 .84489 -1.875 0 Mabsadd m
.72381 .84489 0 2.5 Mabsadd L
.72381 .84489 1.875 0 Mabsadd L
.72381 .84489 0 -2.5 Mabsadd L
.72381 .84489 -1.875 0 Mabsadd L closepath F
.72476 .845 -1.875 0 Mabsadd m
.72476 .845 0 2.5 Mabsadd L
.72476 .845 1.875 0 Mabsadd L
.72476 .845 0 -2.5 Mabsadd L
.72476 .845 -1.875 0 Mabsadd L closepath F
.72571 .84505 -1.875 0 Mabsadd m
.72571 .84505 0 2.5 Mabsadd L
.72571 .84505 1.875 0 Mabsadd L
.72571 .84505 0 -2.5 Mabsadd L
.72571 .84505 -1.875 0 Mabsadd L closepath F
.72667 .84506 -1.875 0 Mabsadd m
.72667 .84506 0 2.5 Mabsadd L
.72667 .84506 1.875 0 Mabsadd L
.72667 .84506 0 -2.5 Mabsadd L
.72667 .84506 -1.875 0 Mabsadd L closepath F
.72762 .84506 -1.875 0 Mabsadd m
.72762 .84506 0 2.5 Mabsadd L
.72762 .84506 1.875 0 Mabsadd L
.72762 .84506 0 -2.5 Mabsadd L
.72762 .84506 -1.875 0 Mabsadd L closepath F
.72857 .84506 -1.875 0 Mabsadd m
.72857 .84506 0 2.5 Mabsadd L
.72857 .84506 1.875 0 Mabsadd L
.72857 .84506 0 -2.5 Mabsadd L
.72857 .84506 -1.875 0 Mabsadd L closepath F
.72952 .84507 -1.875 0 Mabsadd m
.72952 .84507 0 2.5 Mabsadd L
.72952 .84507 1.875 0 Mabsadd L
.72952 .84507 0 -2.5 Mabsadd L
.72952 .84507 -1.875 0 Mabsadd L closepath F
.73048 .84506 -1.875 0 Mabsadd m
.73048 .84506 0 2.5 Mabsadd L
.73048 .84506 1.875 0 Mabsadd L
.73048 .84506 0 -2.5 Mabsadd L
.73048 .84506 -1.875 0 Mabsadd L closepath F
.73143 .84506 -1.875 0 Mabsadd m
.73143 .84506 0 2.5 Mabsadd L
.73143 .84506 1.875 0 Mabsadd L
.73143 .84506 0 -2.5 Mabsadd L
.73143 .84506 -1.875 0 Mabsadd L closepath F
.73238 .84506 -1.875 0 Mabsadd m
.73238 .84506 0 2.5 Mabsadd L
.73238 .84506 1.875 0 Mabsadd L
.73238 .84506 0 -2.5 Mabsadd L
.73238 .84506 -1.875 0 Mabsadd L closepath F
.73333 .84506 -1.875 0 Mabsadd m
.73333 .84506 0 2.5 Mabsadd L
.73333 .84506 1.875 0 Mabsadd L
.73333 .84506 0 -2.5 Mabsadd L
.73333 .84506 -1.875 0 Mabsadd L closepath F
.73429 .84506 -1.875 0 Mabsadd m
.73429 .84506 0 2.5 Mabsadd L
.73429 .84506 1.875 0 Mabsadd L
.73429 .84506 0 -2.5 Mabsadd L
.73429 .84506 -1.875 0 Mabsadd L closepath F
.73524 .84507 -1.875 0 Mabsadd m
.73524 .84507 0 2.5 Mabsadd L
.73524 .84507 1.875 0 Mabsadd L
.73524 .84507 0 -2.5 Mabsadd L
.73524 .84507 -1.875 0 Mabsadd L closepath F
.73619 .84507 -1.875 0 Mabsadd m
.73619 .84507 0 2.5 Mabsadd L
.73619 .84507 1.875 0 Mabsadd L
.73619 .84507 0 -2.5 Mabsadd L
.73619 .84507 -1.875 0 Mabsadd L closepath F
.73714 .84506 -1.875 0 Mabsadd m
.73714 .84506 0 2.5 Mabsadd L
.73714 .84506 1.875 0 Mabsadd L
.73714 .84506 0 -2.5 Mabsadd L
.73714 .84506 -1.875 0 Mabsadd L closepath F
.7381 .84506 -1.875 0 Mabsadd m
.7381 .84506 0 2.5 Mabsadd L
.7381 .84506 1.875 0 Mabsadd L
.7381 .84506 0 -2.5 Mabsadd L
.7381 .84506 -1.875 0 Mabsadd L closepath F
.73905 .84506 -1.875 0 Mabsadd m
.73905 .84506 0 2.5 Mabsadd L
.73905 .84506 1.875 0 Mabsadd L
.73905 .84506 0 -2.5 Mabsadd L
.73905 .84506 -1.875 0 Mabsadd L closepath F
.74 .84506 -1.875 0 Mabsadd m
.74 .84506 0 2.5 Mabsadd L
.74 .84506 1.875 0 Mabsadd L
.74 .84506 0 -2.5 Mabsadd L
.74 .84506 -1.875 0 Mabsadd L closepath F
.74095 .84507 -1.875 0 Mabsadd m
.74095 .84507 0 2.5 Mabsadd L
.74095 .84507 1.875 0 Mabsadd L
.74095 .84507 0 -2.5 Mabsadd L
.74095 .84507 -1.875 0 Mabsadd L closepath F
.7419 .84506 -1.875 0 Mabsadd m
.7419 .84506 0 2.5 Mabsadd L
.7419 .84506 1.875 0 Mabsadd L
.7419 .84506 0 -2.5 Mabsadd L
.7419 .84506 -1.875 0 Mabsadd L closepath F
.74286 .84507 -1.875 0 Mabsadd m
.74286 .84507 0 2.5 Mabsadd L
.74286 .84507 1.875 0 Mabsadd L
.74286 .84507 0 -2.5 Mabsadd L
.74286 .84507 -1.875 0 Mabsadd L closepath F
.74381 .84506 -1.875 0 Mabsadd m
.74381 .84506 0 2.5 Mabsadd L
.74381 .84506 1.875 0 Mabsadd L
.74381 .84506 0 -2.5 Mabsadd L
.74381 .84506 -1.875 0 Mabsadd L closepath F
.74476 .84506 -1.875 0 Mabsadd m
.74476 .84506 0 2.5 Mabsadd L
.74476 .84506 1.875 0 Mabsadd L
.74476 .84506 0 -2.5 Mabsadd L
.74476 .84506 -1.875 0 Mabsadd L closepath F
.74571 .84506 -1.875 0 Mabsadd m
.74571 .84506 0 2.5 Mabsadd L
.74571 .84506 1.875 0 Mabsadd L
.74571 .84506 0 -2.5 Mabsadd L
.74571 .84506 -1.875 0 Mabsadd L closepath F
.74667 .84506 -1.875 0 Mabsadd m
.74667 .84506 0 2.5 Mabsadd L
.74667 .84506 1.875 0 Mabsadd L
.74667 .84506 0 -2.5 Mabsadd L
.74667 .84506 -1.875 0 Mabsadd L closepath F
.74762 .84506 -1.875 0 Mabsadd m
.74762 .84506 0 2.5 Mabsadd L
.74762 .84506 1.875 0 Mabsadd L
.74762 .84506 0 -2.5 Mabsadd L
.74762 .84506 -1.875 0 Mabsadd L closepath F
.74857 .84506 -1.875 0 Mabsadd m
.74857 .84506 0 2.5 Mabsadd L
.74857 .84506 1.875 0 Mabsadd L
.74857 .84506 0 -2.5 Mabsadd L
.74857 .84506 -1.875 0 Mabsadd L closepath F
.74952 .84506 -1.875 0 Mabsadd m
.74952 .84506 0 2.5 Mabsadd L
.74952 .84506 1.875 0 Mabsadd L
.74952 .84506 0 -2.5 Mabsadd L
.74952 .84506 -1.875 0 Mabsadd L closepath F
.75048 .84507 -1.875 0 Mabsadd m
.75048 .84507 0 2.5 Mabsadd L
.75048 .84507 1.875 0 Mabsadd L
.75048 .84507 0 -2.5 Mabsadd L
.75048 .84507 -1.875 0 Mabsadd L closepath F
.75143 .84506 -1.875 0 Mabsadd m
.75143 .84506 0 2.5 Mabsadd L
.75143 .84506 1.875 0 Mabsadd L
.75143 .84506 0 -2.5 Mabsadd L
.75143 .84506 -1.875 0 Mabsadd L closepath F
.75238 .84507 -1.875 0 Mabsadd m
.75238 .84507 0 2.5 Mabsadd L
.75238 .84507 1.875 0 Mabsadd L
.75238 .84507 0 -2.5 Mabsadd L
.75238 .84507 -1.875 0 Mabsadd L closepath F
.75333 .84507 -1.875 0 Mabsadd m
.75333 .84507 0 2.5 Mabsadd L
.75333 .84507 1.875 0 Mabsadd L
.75333 .84507 0 -2.5 Mabsadd L
.75333 .84507 -1.875 0 Mabsadd L closepath F
.75429 .84507 -1.875 0 Mabsadd m
.75429 .84507 0 2.5 Mabsadd L
.75429 .84507 1.875 0 Mabsadd L
.75429 .84507 0 -2.5 Mabsadd L
.75429 .84507 -1.875 0 Mabsadd L closepath F
.75524 .84507 -1.875 0 Mabsadd m
.75524 .84507 0 2.5 Mabsadd L
.75524 .84507 1.875 0 Mabsadd L
.75524 .84507 0 -2.5 Mabsadd L
.75524 .84507 -1.875 0 Mabsadd L closepath F
.75619 .84506 -1.875 0 Mabsadd m
.75619 .84506 0 2.5 Mabsadd L
.75619 .84506 1.875 0 Mabsadd L
.75619 .84506 0 -2.5 Mabsadd L
.75619 .84506 -1.875 0 Mabsadd L closepath F
.75714 .84506 -1.875 0 Mabsadd m
.75714 .84506 0 2.5 Mabsadd L
.75714 .84506 1.875 0 Mabsadd L
.75714 .84506 0 -2.5 Mabsadd L
.75714 .84506 -1.875 0 Mabsadd L closepath F
.7581 .84506 -1.875 0 Mabsadd m
.7581 .84506 0 2.5 Mabsadd L
.7581 .84506 1.875 0 Mabsadd L
.7581 .84506 0 -2.5 Mabsadd L
.7581 .84506 -1.875 0 Mabsadd L closepath F
.75905 .84506 -1.875 0 Mabsadd m
.75905 .84506 0 2.5 Mabsadd L
.75905 .84506 1.875 0 Mabsadd L
.75905 .84506 0 -2.5 Mabsadd L
.75905 .84506 -1.875 0 Mabsadd L closepath F
.76 .84506 -1.875 0 Mabsadd m
.76 .84506 0 2.5 Mabsadd L
.76 .84506 1.875 0 Mabsadd L
.76 .84506 0 -2.5 Mabsadd L
.76 .84506 -1.875 0 Mabsadd L closepath F
.76095 .84506 -1.875 0 Mabsadd m
.76095 .84506 0 2.5 Mabsadd L
.76095 .84506 1.875 0 Mabsadd L
.76095 .84506 0 -2.5 Mabsadd L
.76095 .84506 -1.875 0 Mabsadd L closepath F
.7619 .84507 -1.875 0 Mabsadd m
.7619 .84507 0 2.5 Mabsadd L
.7619 .84507 1.875 0 Mabsadd L
.7619 .84507 0 -2.5 Mabsadd L
.7619 .84507 -1.875 0 Mabsadd L closepath F
.76286 .84506 -1.875 0 Mabsadd m
.76286 .84506 0 2.5 Mabsadd L
.76286 .84506 1.875 0 Mabsadd L
.76286 .84506 0 -2.5 Mabsadd L
.76286 .84506 -1.875 0 Mabsadd L closepath F
.76381 .84506 -1.875 0 Mabsadd m
.76381 .84506 0 2.5 Mabsadd L
.76381 .84506 1.875 0 Mabsadd L
.76381 .84506 0 -2.5 Mabsadd L
.76381 .84506 -1.875 0 Mabsadd L closepath F
.76476 .84505 -1.875 0 Mabsadd m
.76476 .84505 0 2.5 Mabsadd L
.76476 .84505 1.875 0 Mabsadd L
.76476 .84505 0 -2.5 Mabsadd L
.76476 .84505 -1.875 0 Mabsadd L closepath F
.76571 .84502 -1.875 0 Mabsadd m
.76571 .84502 0 2.5 Mabsadd L
.76571 .84502 1.875 0 Mabsadd L
.76571 .84502 0 -2.5 Mabsadd L
.76571 .84502 -1.875 0 Mabsadd L closepath F
.76667 .84495 -1.875 0 Mabsadd m
.76667 .84495 0 2.5 Mabsadd L
.76667 .84495 1.875 0 Mabsadd L
.76667 .84495 0 -2.5 Mabsadd L
.76667 .84495 -1.875 0 Mabsadd L closepath F
.76762 .8448 -1.875 0 Mabsadd m
.76762 .8448 0 2.5 Mabsadd L
.76762 .8448 1.875 0 Mabsadd L
.76762 .8448 0 -2.5 Mabsadd L
.76762 .8448 -1.875 0 Mabsadd L closepath F
.76857 .84452 -1.875 0 Mabsadd m
.76857 .84452 0 2.5 Mabsadd L
.76857 .84452 1.875 0 Mabsadd L
.76857 .84452 0 -2.5 Mabsadd L
.76857 .84452 -1.875 0 Mabsadd L closepath F
.76952 .83504 -1.875 0 Mabsadd m
.76952 .83504 0 2.5 Mabsadd L
.76952 .83504 1.875 0 Mabsadd L
.76952 .83504 0 -2.5 Mabsadd L
.76952 .83504 -1.875 0 Mabsadd L closepath F
.77048 .7444 -1.875 0 Mabsadd m
.77048 .7444 0 2.5 Mabsadd L
.77048 .7444 1.875 0 Mabsadd L
.77048 .7444 0 -2.5 Mabsadd L
.77048 .7444 -1.875 0 Mabsadd L closepath F
.77143 .81361 -1.875 0 Mabsadd m
.77143 .81361 0 2.5 Mabsadd L
.77143 .81361 1.875 0 Mabsadd L
.77143 .81361 0 -2.5 Mabsadd L
.77143 .81361 -1.875 0 Mabsadd L closepath F
.77238 .2006 -1.875 0 Mabsadd m
.77238 .2006 0 2.5 Mabsadd L
.77238 .2006 1.875 0 Mabsadd L
.77238 .2006 0 -2.5 Mabsadd L
.77238 .2006 -1.875 0 Mabsadd L closepath F
.77333 .65759 -1.875 0 Mabsadd m
.77333 .65759 0 2.5 Mabsadd L
.77333 .65759 1.875 0 Mabsadd L
.77333 .65759 0 -2.5 Mabsadd L
.77333 .65759 -1.875 0 Mabsadd L closepath F
.77429 .69483 -1.875 0 Mabsadd m
.77429 .69483 0 2.5 Mabsadd L
.77429 .69483 1.875 0 Mabsadd L
.77429 .69483 0 -2.5 Mabsadd L
.77429 .69483 -1.875 0 Mabsadd L closepath F
.77524 .74506 -1.875 0 Mabsadd m
.77524 .74506 0 2.5 Mabsadd L
.77524 .74506 1.875 0 Mabsadd L
.77524 .74506 0 -2.5 Mabsadd L
.77524 .74506 -1.875 0 Mabsadd L closepath F
.77619 .74766 -1.875 0 Mabsadd m
.77619 .74766 0 2.5 Mabsadd L
.77619 .74766 1.875 0 Mabsadd L
.77619 .74766 0 -2.5 Mabsadd L
.77619 .74766 -1.875 0 Mabsadd L closepath F
.77714 .74783 -1.875 0 Mabsadd m
.77714 .74783 0 2.5 Mabsadd L
.77714 .74783 1.875 0 Mabsadd L
.77714 .74783 0 -2.5 Mabsadd L
.77714 .74783 -1.875 0 Mabsadd L closepath F
.7781 .74796 -1.875 0 Mabsadd m
.7781 .74796 0 2.5 Mabsadd L
.7781 .74796 1.875 0 Mabsadd L
.7781 .74796 0 -2.5 Mabsadd L
.7781 .74796 -1.875 0 Mabsadd L closepath F
.77905 .748 -1.875 0 Mabsadd m
.77905 .748 0 2.5 Mabsadd L
.77905 .748 1.875 0 Mabsadd L
.77905 .748 0 -2.5 Mabsadd L
.77905 .748 -1.875 0 Mabsadd L closepath F
.78 .74802 -1.875 0 Mabsadd m
.78 .74802 0 2.5 Mabsadd L
.78 .74802 1.875 0 Mabsadd L
.78 .74802 0 -2.5 Mabsadd L
.78 .74802 -1.875 0 Mabsadd L closepath F
.78095 .74803 -1.875 0 Mabsadd m
.78095 .74803 0 2.5 Mabsadd L
.78095 .74803 1.875 0 Mabsadd L
.78095 .74803 0 -2.5 Mabsadd L
.78095 .74803 -1.875 0 Mabsadd L closepath F
.7819 .74803 -1.875 0 Mabsadd m
.7819 .74803 0 2.5 Mabsadd L
.7819 .74803 1.875 0 Mabsadd L
.7819 .74803 0 -2.5 Mabsadd L
.7819 .74803 -1.875 0 Mabsadd L closepath F
.78286 .74803 -1.875 0 Mabsadd m
.78286 .74803 0 2.5 Mabsadd L
.78286 .74803 1.875 0 Mabsadd L
.78286 .74803 0 -2.5 Mabsadd L
.78286 .74803 -1.875 0 Mabsadd L closepath F
.78381 .74803 -1.875 0 Mabsadd m
.78381 .74803 0 2.5 Mabsadd L
.78381 .74803 1.875 0 Mabsadd L
.78381 .74803 0 -2.5 Mabsadd L
.78381 .74803 -1.875 0 Mabsadd L closepath F
.78476 .74803 -1.875 0 Mabsadd m
.78476 .74803 0 2.5 Mabsadd L
.78476 .74803 1.875 0 Mabsadd L
.78476 .74803 0 -2.5 Mabsadd L
.78476 .74803 -1.875 0 Mabsadd L closepath F
.78571 .74803 -1.875 0 Mabsadd m
.78571 .74803 0 2.5 Mabsadd L
.78571 .74803 1.875 0 Mabsadd L
.78571 .74803 0 -2.5 Mabsadd L
.78571 .74803 -1.875 0 Mabsadd L closepath F
.78667 .74803 -1.875 0 Mabsadd m
.78667 .74803 0 2.5 Mabsadd L
.78667 .74803 1.875 0 Mabsadd L
.78667 .74803 0 -2.5 Mabsadd L
.78667 .74803 -1.875 0 Mabsadd L closepath F
.78762 .74803 -1.875 0 Mabsadd m
.78762 .74803 0 2.5 Mabsadd L
.78762 .74803 1.875 0 Mabsadd L
.78762 .74803 0 -2.5 Mabsadd L
.78762 .74803 -1.875 0 Mabsadd L closepath F
.78857 .74803 -1.875 0 Mabsadd m
.78857 .74803 0 2.5 Mabsadd L
.78857 .74803 1.875 0 Mabsadd L
.78857 .74803 0 -2.5 Mabsadd L
.78857 .74803 -1.875 0 Mabsadd L closepath F
.78952 .74803 -1.875 0 Mabsadd m
.78952 .74803 0 2.5 Mabsadd L
.78952 .74803 1.875 0 Mabsadd L
.78952 .74803 0 -2.5 Mabsadd L
.78952 .74803 -1.875 0 Mabsadd L closepath F
.79048 .74803 -1.875 0 Mabsadd m
.79048 .74803 0 2.5 Mabsadd L
.79048 .74803 1.875 0 Mabsadd L
.79048 .74803 0 -2.5 Mabsadd L
.79048 .74803 -1.875 0 Mabsadd L closepath F
.79143 .74803 -1.875 0 Mabsadd m
.79143 .74803 0 2.5 Mabsadd L
.79143 .74803 1.875 0 Mabsadd L
.79143 .74803 0 -2.5 Mabsadd L
.79143 .74803 -1.875 0 Mabsadd L closepath F
.79238 .74803 -1.875 0 Mabsadd m
.79238 .74803 0 2.5 Mabsadd L
.79238 .74803 1.875 0 Mabsadd L
.79238 .74803 0 -2.5 Mabsadd L
.79238 .74803 -1.875 0 Mabsadd L closepath F
.79333 .74803 -1.875 0 Mabsadd m
.79333 .74803 0 2.5 Mabsadd L
.79333 .74803 1.875 0 Mabsadd L
.79333 .74803 0 -2.5 Mabsadd L
.79333 .74803 -1.875 0 Mabsadd L closepath F
.79429 .74803 -1.875 0 Mabsadd m
.79429 .74803 0 2.5 Mabsadd L
.79429 .74803 1.875 0 Mabsadd L
.79429 .74803 0 -2.5 Mabsadd L
.79429 .74803 -1.875 0 Mabsadd L closepath F
.79524 .74803 -1.875 0 Mabsadd m
.79524 .74803 0 2.5 Mabsadd L
.79524 .74803 1.875 0 Mabsadd L
.79524 .74803 0 -2.5 Mabsadd L
.79524 .74803 -1.875 0 Mabsadd L closepath F
.79619 .74803 -1.875 0 Mabsadd m
.79619 .74803 0 2.5 Mabsadd L
.79619 .74803 1.875 0 Mabsadd L
.79619 .74803 0 -2.5 Mabsadd L
.79619 .74803 -1.875 0 Mabsadd L closepath F
.79714 .74803 -1.875 0 Mabsadd m
.79714 .74803 0 2.5 Mabsadd L
.79714 .74803 1.875 0 Mabsadd L
.79714 .74803 0 -2.5 Mabsadd L
.79714 .74803 -1.875 0 Mabsadd L closepath F
.7981 .74803 -1.875 0 Mabsadd m
.7981 .74803 0 2.5 Mabsadd L
.7981 .74803 1.875 0 Mabsadd L
.7981 .74803 0 -2.5 Mabsadd L
.7981 .74803 -1.875 0 Mabsadd L closepath F
.79905 .74803 -1.875 0 Mabsadd m
.79905 .74803 0 2.5 Mabsadd L
.79905 .74803 1.875 0 Mabsadd L
.79905 .74803 0 -2.5 Mabsadd L
.79905 .74803 -1.875 0 Mabsadd L closepath F
.8 .74803 -1.875 0 Mabsadd m
.8 .74803 0 2.5 Mabsadd L
.8 .74803 1.875 0 Mabsadd L
.8 .74803 0 -2.5 Mabsadd L
.8 .74803 -1.875 0 Mabsadd L closepath F
.80095 .74803 -1.875 0 Mabsadd m
.80095 .74803 0 2.5 Mabsadd L
.80095 .74803 1.875 0 Mabsadd L
.80095 .74803 0 -2.5 Mabsadd L
.80095 .74803 -1.875 0 Mabsadd L closepath F
.8019 .74803 -1.875 0 Mabsadd m
.8019 .74803 0 2.5 Mabsadd L
.8019 .74803 1.875 0 Mabsadd L
.8019 .74803 0 -2.5 Mabsadd L
.8019 .74803 -1.875 0 Mabsadd L closepath F
.80286 .74803 -1.875 0 Mabsadd m
.80286 .74803 0 2.5 Mabsadd L
.80286 .74803 1.875 0 Mabsadd L
.80286 .74803 0 -2.5 Mabsadd L
.80286 .74803 -1.875 0 Mabsadd L closepath F
.80381 .74803 -1.875 0 Mabsadd m
.80381 .74803 0 2.5 Mabsadd L
.80381 .74803 1.875 0 Mabsadd L
.80381 .74803 0 -2.5 Mabsadd L
.80381 .74803 -1.875 0 Mabsadd L closepath F
.80476 .74803 -1.875 0 Mabsadd m
.80476 .74803 0 2.5 Mabsadd L
.80476 .74803 1.875 0 Mabsadd L
.80476 .74803 0 -2.5 Mabsadd L
.80476 .74803 -1.875 0 Mabsadd L closepath F
.80571 .74803 -1.875 0 Mabsadd m
.80571 .74803 0 2.5 Mabsadd L
.80571 .74803 1.875 0 Mabsadd L
.80571 .74803 0 -2.5 Mabsadd L
.80571 .74803 -1.875 0 Mabsadd L closepath F
.80667 .74803 -1.875 0 Mabsadd m
.80667 .74803 0 2.5 Mabsadd L
.80667 .74803 1.875 0 Mabsadd L
.80667 .74803 0 -2.5 Mabsadd L
.80667 .74803 -1.875 0 Mabsadd L closepath F
.80762 .74803 -1.875 0 Mabsadd m
.80762 .74803 0 2.5 Mabsadd L
.80762 .74803 1.875 0 Mabsadd L
.80762 .74803 0 -2.5 Mabsadd L
.80762 .74803 -1.875 0 Mabsadd L closepath F
.80857 .74803 -1.875 0 Mabsadd m
.80857 .74803 0 2.5 Mabsadd L
.80857 .74803 1.875 0 Mabsadd L
.80857 .74803 0 -2.5 Mabsadd L
.80857 .74803 -1.875 0 Mabsadd L closepath F
.80952 .74803 -1.875 0 Mabsadd m
.80952 .74803 0 2.5 Mabsadd L
.80952 .74803 1.875 0 Mabsadd L
.80952 .74803 0 -2.5 Mabsadd L
.80952 .74803 -1.875 0 Mabsadd L closepath F
.81048 .74803 -1.875 0 Mabsadd m
.81048 .74803 0 2.5 Mabsadd L
.81048 .74803 1.875 0 Mabsadd L
.81048 .74803 0 -2.5 Mabsadd L
.81048 .74803 -1.875 0 Mabsadd L closepath F
.81143 .74803 -1.875 0 Mabsadd m
.81143 .74803 0 2.5 Mabsadd L
.81143 .74803 1.875 0 Mabsadd L
.81143 .74803 0 -2.5 Mabsadd L
.81143 .74803 -1.875 0 Mabsadd L closepath F
.81238 .74803 -1.875 0 Mabsadd m
.81238 .74803 0 2.5 Mabsadd L
.81238 .74803 1.875 0 Mabsadd L
.81238 .74803 0 -2.5 Mabsadd L
.81238 .74803 -1.875 0 Mabsadd L closepath F
.81333 .74803 -1.875 0 Mabsadd m
.81333 .74803 0 2.5 Mabsadd L
.81333 .74803 1.875 0 Mabsadd L
.81333 .74803 0 -2.5 Mabsadd L
.81333 .74803 -1.875 0 Mabsadd L closepath F
.81429 .74803 -1.875 0 Mabsadd m
.81429 .74803 0 2.5 Mabsadd L
.81429 .74803 1.875 0 Mabsadd L
.81429 .74803 0 -2.5 Mabsadd L
.81429 .74803 -1.875 0 Mabsadd L closepath F
.81524 .74803 -1.875 0 Mabsadd m
.81524 .74803 0 2.5 Mabsadd L
.81524 .74803 1.875 0 Mabsadd L
.81524 .74803 0 -2.5 Mabsadd L
.81524 .74803 -1.875 0 Mabsadd L closepath F
.81619 .74803 -1.875 0 Mabsadd m
.81619 .74803 0 2.5 Mabsadd L
.81619 .74803 1.875 0 Mabsadd L
.81619 .74803 0 -2.5 Mabsadd L
.81619 .74803 -1.875 0 Mabsadd L closepath F
.81714 .74803 -1.875 0 Mabsadd m
.81714 .74803 0 2.5 Mabsadd L
.81714 .74803 1.875 0 Mabsadd L
.81714 .74803 0 -2.5 Mabsadd L
.81714 .74803 -1.875 0 Mabsadd L closepath F
.8181 .74802 -1.875 0 Mabsadd m
.8181 .74802 0 2.5 Mabsadd L
.8181 .74802 1.875 0 Mabsadd L
.8181 .74802 0 -2.5 Mabsadd L
.8181 .74802 -1.875 0 Mabsadd L closepath F
.81905 .748 -1.875 0 Mabsadd m
.81905 .748 0 2.5 Mabsadd L
.81905 .748 1.875 0 Mabsadd L
.81905 .748 0 -2.5 Mabsadd L
.81905 .748 -1.875 0 Mabsadd L closepath F
.82 .74794 -1.875 0 Mabsadd m
.82 .74794 0 2.5 Mabsadd L
.82 .74794 1.875 0 Mabsadd L
.82 .74794 0 -2.5 Mabsadd L
.82 .74794 -1.875 0 Mabsadd L closepath F
.82095 .7478 -1.875 0 Mabsadd m
.82095 .7478 0 2.5 Mabsadd L
.82095 .7478 1.875 0 Mabsadd L
.82095 .7478 0 -2.5 Mabsadd L
.82095 .7478 -1.875 0 Mabsadd L closepath F
.8219 .74757 -1.875 0 Mabsadd m
.8219 .74757 0 2.5 Mabsadd L
.8219 .74757 1.875 0 Mabsadd L
.8219 .74757 0 -2.5 Mabsadd L
.8219 .74757 -1.875 0 Mabsadd L closepath F
.82286 .74139 -1.875 0 Mabsadd m
.82286 .74139 0 2.5 Mabsadd L
.82286 .74139 1.875 0 Mabsadd L
.82286 .74139 0 -2.5 Mabsadd L
.82286 .74139 -1.875 0 Mabsadd L closepath F
.82381 .66404 -1.875 0 Mabsadd m
.82381 .66404 0 2.5 Mabsadd L
.82381 .66404 1.875 0 Mabsadd L
.82381 .66404 0 -2.5 Mabsadd L
.82381 .66404 -1.875 0 Mabsadd L closepath F
.82476 .70785 -1.875 0 Mabsadd m
.82476 .70785 0 2.5 Mabsadd L
.82476 .70785 1.875 0 Mabsadd L
.82476 .70785 0 -2.5 Mabsadd L
.82476 .70785 -1.875 0 Mabsadd L closepath F
.82571 .04169 -1.875 0 Mabsadd m
.82571 .04169 0 2.5 Mabsadd L
.82571 .04169 1.875 0 Mabsadd L
.82571 .04169 0 -2.5 Mabsadd L
.82571 .04169 -1.875 0 Mabsadd L closepath F
.82667 .64836 -1.875 0 Mabsadd m
.82667 .64836 0 2.5 Mabsadd L
.82667 .64836 1.875 0 Mabsadd L
.82667 .64836 0 -2.5 Mabsadd L
.82667 .64836 -1.875 0 Mabsadd L closepath F
.82762 .64195 -1.875 0 Mabsadd m
.82762 .64195 0 2.5 Mabsadd L
.82762 .64195 1.875 0 Mabsadd L
.82762 .64195 0 -2.5 Mabsadd L
.82762 .64195 -1.875 0 Mabsadd L closepath F
.82857 .70629 -1.875 0 Mabsadd m
.82857 .70629 0 2.5 Mabsadd L
.82857 .70629 1.875 0 Mabsadd L
.82857 .70629 0 -2.5 Mabsadd L
.82857 .70629 -1.875 0 Mabsadd L closepath F
.82952 .71045 -1.875 0 Mabsadd m
.82952 .71045 0 2.5 Mabsadd L
.82952 .71045 1.875 0 Mabsadd L
.82952 .71045 0 -2.5 Mabsadd L
.82952 .71045 -1.875 0 Mabsadd L closepath F
.83048 .71064 -1.875 0 Mabsadd m
.83048 .71064 0 2.5 Mabsadd L
.83048 .71064 1.875 0 Mabsadd L
.83048 .71064 0 -2.5 Mabsadd L
.83048 .71064 -1.875 0 Mabsadd L closepath F
.83143 .71077 -1.875 0 Mabsadd m
.83143 .71077 0 2.5 Mabsadd L
.83143 .71077 1.875 0 Mabsadd L
.83143 .71077 0 -2.5 Mabsadd L
.83143 .71077 -1.875 0 Mabsadd L closepath F
.83238 .71083 -1.875 0 Mabsadd m
.83238 .71083 0 2.5 Mabsadd L
.83238 .71083 1.875 0 Mabsadd L
.83238 .71083 0 -2.5 Mabsadd L
.83238 .71083 -1.875 0 Mabsadd L closepath F
.83333 .71084 -1.875 0 Mabsadd m
.83333 .71084 0 2.5 Mabsadd L
.83333 .71084 1.875 0 Mabsadd L
.83333 .71084 0 -2.5 Mabsadd L
.83333 .71084 -1.875 0 Mabsadd L closepath F
.83429 .71085 -1.875 0 Mabsadd m
.83429 .71085 0 2.5 Mabsadd L
.83429 .71085 1.875 0 Mabsadd L
.83429 .71085 0 -2.5 Mabsadd L
.83429 .71085 -1.875 0 Mabsadd L closepath F
.83524 .71085 -1.875 0 Mabsadd m
.83524 .71085 0 2.5 Mabsadd L
.83524 .71085 1.875 0 Mabsadd L
.83524 .71085 0 -2.5 Mabsadd L
.83524 .71085 -1.875 0 Mabsadd L closepath F
.83619 .71085 -1.875 0 Mabsadd m
.83619 .71085 0 2.5 Mabsadd L
.83619 .71085 1.875 0 Mabsadd L
.83619 .71085 0 -2.5 Mabsadd L
.83619 .71085 -1.875 0 Mabsadd L closepath F
.83714 .71085 -1.875 0 Mabsadd m
.83714 .71085 0 2.5 Mabsadd L
.83714 .71085 1.875 0 Mabsadd L
.83714 .71085 0 -2.5 Mabsadd L
.83714 .71085 -1.875 0 Mabsadd L closepath F
.8381 .71086 -1.875 0 Mabsadd m
.8381 .71086 0 2.5 Mabsadd L
.8381 .71086 1.875 0 Mabsadd L
.8381 .71086 0 -2.5 Mabsadd L
.8381 .71086 -1.875 0 Mabsadd L closepath F
.83905 .71085 -1.875 0 Mabsadd m
.83905 .71085 0 2.5 Mabsadd L
.83905 .71085 1.875 0 Mabsadd L
.83905 .71085 0 -2.5 Mabsadd L
.83905 .71085 -1.875 0 Mabsadd L closepath F
.84 .71086 -1.875 0 Mabsadd m
.84 .71086 0 2.5 Mabsadd L
.84 .71086 1.875 0 Mabsadd L
.84 .71086 0 -2.5 Mabsadd L
.84 .71086 -1.875 0 Mabsadd L closepath F
.84095 .71086 -1.875 0 Mabsadd m
.84095 .71086 0 2.5 Mabsadd L
.84095 .71086 1.875 0 Mabsadd L
.84095 .71086 0 -2.5 Mabsadd L
.84095 .71086 -1.875 0 Mabsadd L closepath F
.8419 .71086 -1.875 0 Mabsadd m
.8419 .71086 0 2.5 Mabsadd L
.8419 .71086 1.875 0 Mabsadd L
.8419 .71086 0 -2.5 Mabsadd L
.8419 .71086 -1.875 0 Mabsadd L closepath F
.84286 .71086 -1.875 0 Mabsadd m
.84286 .71086 0 2.5 Mabsadd L
.84286 .71086 1.875 0 Mabsadd L
.84286 .71086 0 -2.5 Mabsadd L
.84286 .71086 -1.875 0 Mabsadd L closepath F
.84381 .71085 -1.875 0 Mabsadd m
.84381 .71085 0 2.5 Mabsadd L
.84381 .71085 1.875 0 Mabsadd L
.84381 .71085 0 -2.5 Mabsadd L
.84381 .71085 -1.875 0 Mabsadd L closepath F
.84476 .71086 -1.875 0 Mabsadd m
.84476 .71086 0 2.5 Mabsadd L
.84476 .71086 1.875 0 Mabsadd L
.84476 .71086 0 -2.5 Mabsadd L
.84476 .71086 -1.875 0 Mabsadd L closepath F
.84571 .71086 -1.875 0 Mabsadd m
.84571 .71086 0 2.5 Mabsadd L
.84571 .71086 1.875 0 Mabsadd L
.84571 .71086 0 -2.5 Mabsadd L
.84571 .71086 -1.875 0 Mabsadd L closepath F
.84667 .71085 -1.875 0 Mabsadd m
.84667 .71085 0 2.5 Mabsadd L
.84667 .71085 1.875 0 Mabsadd L
.84667 .71085 0 -2.5 Mabsadd L
.84667 .71085 -1.875 0 Mabsadd L closepath F
.84762 .71085 -1.875 0 Mabsadd m
.84762 .71085 0 2.5 Mabsadd L
.84762 .71085 1.875 0 Mabsadd L
.84762 .71085 0 -2.5 Mabsadd L
.84762 .71085 -1.875 0 Mabsadd L closepath F
.84857 .71085 -1.875 0 Mabsadd m
.84857 .71085 0 2.5 Mabsadd L
.84857 .71085 1.875 0 Mabsadd L
.84857 .71085 0 -2.5 Mabsadd L
.84857 .71085 -1.875 0 Mabsadd L closepath F
.84952 .71086 -1.875 0 Mabsadd m
.84952 .71086 0 2.5 Mabsadd L
.84952 .71086 1.875 0 Mabsadd L
.84952 .71086 0 -2.5 Mabsadd L
.84952 .71086 -1.875 0 Mabsadd L closepath F
.85048 .71085 -1.875 0 Mabsadd m
.85048 .71085 0 2.5 Mabsadd L
.85048 .71085 1.875 0 Mabsadd L
.85048 .71085 0 -2.5 Mabsadd L
.85048 .71085 -1.875 0 Mabsadd L closepath F
.85143 .71086 -1.875 0 Mabsadd m
.85143 .71086 0 2.5 Mabsadd L
.85143 .71086 1.875 0 Mabsadd L
.85143 .71086 0 -2.5 Mabsadd L
.85143 .71086 -1.875 0 Mabsadd L closepath F
.85238 .71086 -1.875 0 Mabsadd m
.85238 .71086 0 2.5 Mabsadd L
.85238 .71086 1.875 0 Mabsadd L
.85238 .71086 0 -2.5 Mabsadd L
.85238 .71086 -1.875 0 Mabsadd L closepath F
.85333 .71085 -1.875 0 Mabsadd m
.85333 .71085 0 2.5 Mabsadd L
.85333 .71085 1.875 0 Mabsadd L
.85333 .71085 0 -2.5 Mabsadd L
.85333 .71085 -1.875 0 Mabsadd L closepath F
.85429 .71086 -1.875 0 Mabsadd m
.85429 .71086 0 2.5 Mabsadd L
.85429 .71086 1.875 0 Mabsadd L
.85429 .71086 0 -2.5 Mabsadd L
.85429 .71086 -1.875 0 Mabsadd L closepath F
.85524 .71085 -1.875 0 Mabsadd m
.85524 .71085 0 2.5 Mabsadd L
.85524 .71085 1.875 0 Mabsadd L
.85524 .71085 0 -2.5 Mabsadd L
.85524 .71085 -1.875 0 Mabsadd L closepath F
.85619 .71085 -1.875 0 Mabsadd m
.85619 .71085 0 2.5 Mabsadd L
.85619 .71085 1.875 0 Mabsadd L
.85619 .71085 0 -2.5 Mabsadd L
.85619 .71085 -1.875 0 Mabsadd L closepath F
.85714 .71085 -1.875 0 Mabsadd m
.85714 .71085 0 2.5 Mabsadd L
.85714 .71085 1.875 0 Mabsadd L
.85714 .71085 0 -2.5 Mabsadd L
.85714 .71085 -1.875 0 Mabsadd L closepath F
.8581 .71085 -1.875 0 Mabsadd m
.8581 .71085 0 2.5 Mabsadd L
.8581 .71085 1.875 0 Mabsadd L
.8581 .71085 0 -2.5 Mabsadd L
.8581 .71085 -1.875 0 Mabsadd L closepath F
.85905 .71086 -1.875 0 Mabsadd m
.85905 .71086 0 2.5 Mabsadd L
.85905 .71086 1.875 0 Mabsadd L
.85905 .71086 0 -2.5 Mabsadd L
.85905 .71086 -1.875 0 Mabsadd L closepath F
.86 .71086 -1.875 0 Mabsadd m
.86 .71086 0 2.5 Mabsadd L
.86 .71086 1.875 0 Mabsadd L
.86 .71086 0 -2.5 Mabsadd L
.86 .71086 -1.875 0 Mabsadd L closepath F
.86095 .71086 -1.875 0 Mabsadd m
.86095 .71086 0 2.5 Mabsadd L
.86095 .71086 1.875 0 Mabsadd L
.86095 .71086 0 -2.5 Mabsadd L
.86095 .71086 -1.875 0 Mabsadd L closepath F
.8619 .71086 -1.875 0 Mabsadd m
.8619 .71086 0 2.5 Mabsadd L
.8619 .71086 1.875 0 Mabsadd L
.8619 .71086 0 -2.5 Mabsadd L
.8619 .71086 -1.875 0 Mabsadd L closepath F
.86286 .71086 -1.875 0 Mabsadd m
.86286 .71086 0 2.5 Mabsadd L
.86286 .71086 1.875 0 Mabsadd L
.86286 .71086 0 -2.5 Mabsadd L
.86286 .71086 -1.875 0 Mabsadd L closepath F
.86381 .71086 -1.875 0 Mabsadd m
.86381 .71086 0 2.5 Mabsadd L
.86381 .71086 1.875 0 Mabsadd L
.86381 .71086 0 -2.5 Mabsadd L
.86381 .71086 -1.875 0 Mabsadd L closepath F
.86476 .71086 -1.875 0 Mabsadd m
.86476 .71086 0 2.5 Mabsadd L
.86476 .71086 1.875 0 Mabsadd L
.86476 .71086 0 -2.5 Mabsadd L
.86476 .71086 -1.875 0 Mabsadd L closepath F
.86571 .71086 -1.875 0 Mabsadd m
.86571 .71086 0 2.5 Mabsadd L
.86571 .71086 1.875 0 Mabsadd L
.86571 .71086 0 -2.5 Mabsadd L
.86571 .71086 -1.875 0 Mabsadd L closepath F
.86667 .71086 -1.875 0 Mabsadd m
.86667 .71086 0 2.5 Mabsadd L
.86667 .71086 1.875 0 Mabsadd L
.86667 .71086 0 -2.5 Mabsadd L
.86667 .71086 -1.875 0 Mabsadd L closepath F
.86762 .71086 -1.875 0 Mabsadd m
.86762 .71086 0 2.5 Mabsadd L
.86762 .71086 1.875 0 Mabsadd L
.86762 .71086 0 -2.5 Mabsadd L
.86762 .71086 -1.875 0 Mabsadd L closepath F
.86857 .71086 -1.875 0 Mabsadd m
.86857 .71086 0 2.5 Mabsadd L
.86857 .71086 1.875 0 Mabsadd L
.86857 .71086 0 -2.5 Mabsadd L
.86857 .71086 -1.875 0 Mabsadd L closepath F
.86952 .71086 -1.875 0 Mabsadd m
.86952 .71086 0 2.5 Mabsadd L
.86952 .71086 1.875 0 Mabsadd L
.86952 .71086 0 -2.5 Mabsadd L
.86952 .71086 -1.875 0 Mabsadd L closepath F
.87048 .71086 -1.875 0 Mabsadd m
.87048 .71086 0 2.5 Mabsadd L
.87048 .71086 1.875 0 Mabsadd L
.87048 .71086 0 -2.5 Mabsadd L
.87048 .71086 -1.875 0 Mabsadd L closepath F
.87143 .71085 -1.875 0 Mabsadd m
.87143 .71085 0 2.5 Mabsadd L
.87143 .71085 1.875 0 Mabsadd L
.87143 .71085 0 -2.5 Mabsadd L
.87143 .71085 -1.875 0 Mabsadd L closepath F
.87238 .71083 -1.875 0 Mabsadd m
.87238 .71083 0 2.5 Mabsadd L
.87238 .71083 1.875 0 Mabsadd L
.87238 .71083 0 -2.5 Mabsadd L
.87238 .71083 -1.875 0 Mabsadd L closepath F
.87333 .71078 -1.875 0 Mabsadd m
.87333 .71078 0 2.5 Mabsadd L
.87333 .71078 1.875 0 Mabsadd L
.87333 .71078 0 -2.5 Mabsadd L
.87333 .71078 -1.875 0 Mabsadd L closepath F
.87429 .71064 -1.875 0 Mabsadd m
.87429 .71064 0 2.5 Mabsadd L
.87429 .71064 1.875 0 Mabsadd L
.87429 .71064 0 -2.5 Mabsadd L
.87429 .71064 -1.875 0 Mabsadd L closepath F
.87524 .71046 -1.875 0 Mabsadd m
.87524 .71046 0 2.5 Mabsadd L
.87524 .71046 1.875 0 Mabsadd L
.87524 .71046 0 -2.5 Mabsadd L
.87524 .71046 -1.875 0 Mabsadd L closepath F
.87619 .70651 -1.875 0 Mabsadd m
.87619 .70651 0 2.5 Mabsadd L
.87619 .70651 1.875 0 Mabsadd L
.87619 .70651 0 -2.5 Mabsadd L
.87619 .70651 -1.875 0 Mabsadd L closepath F
.87714 .64378 -1.875 0 Mabsadd m
.87714 .64378 0 2.5 Mabsadd L
.87714 .64378 1.875 0 Mabsadd L
.87714 .64378 0 -2.5 Mabsadd L
.87714 .64378 -1.875 0 Mabsadd L closepath F
.8781 .64518 -1.875 0 Mabsadd m
.8781 .64518 0 2.5 Mabsadd L
.8781 .64518 1.875 0 Mabsadd L
.8781 .64518 0 -2.5 Mabsadd L
.8781 .64518 -1.875 0 Mabsadd L closepath F
.87905 .05178 -1.875 0 Mabsadd m
.87905 .05178 0 2.5 Mabsadd L
.87905 .05178 1.875 0 Mabsadd L
.87905 .05178 0 -2.5 Mabsadd L
.87905 .05178 -1.875 0 Mabsadd L closepath F
.88 .71877 -1.875 0 Mabsadd m
.88 .71877 0 2.5 Mabsadd L
.88 .71877 1.875 0 Mabsadd L
.88 .71877 0 -2.5 Mabsadd L
.88 .71877 -1.875 0 Mabsadd L closepath F
.88095 .67092 -1.875 0 Mabsadd m
.88095 .67092 0 2.5 Mabsadd L
.88095 .67092 1.875 0 Mabsadd L
.88095 .67092 0 -2.5 Mabsadd L
.88095 .67092 -1.875 0 Mabsadd L closepath F
.8819 .74986 -1.875 0 Mabsadd m
.8819 .74986 0 2.5 Mabsadd L
.8819 .74986 1.875 0 Mabsadd L
.8819 .74986 0 -2.5 Mabsadd L
.8819 .74986 -1.875 0 Mabsadd L closepath F
.88286 .75636 -1.875 0 Mabsadd m
.88286 .75636 0 2.5 Mabsadd L
.88286 .75636 1.875 0 Mabsadd L
.88286 .75636 0 -2.5 Mabsadd L
.88286 .75636 -1.875 0 Mabsadd L closepath F
.88381 .75658 -1.875 0 Mabsadd m
.88381 .75658 0 2.5 Mabsadd L
.88381 .75658 1.875 0 Mabsadd L
.88381 .75658 0 -2.5 Mabsadd L
.88381 .75658 -1.875 0 Mabsadd L closepath F
.88476 .75672 -1.875 0 Mabsadd m
.88476 .75672 0 2.5 Mabsadd L
.88476 .75672 1.875 0 Mabsadd L
.88476 .75672 0 -2.5 Mabsadd L
.88476 .75672 -1.875 0 Mabsadd L closepath F
.88571 .75679 -1.875 0 Mabsadd m
.88571 .75679 0 2.5 Mabsadd L
.88571 .75679 1.875 0 Mabsadd L
.88571 .75679 0 -2.5 Mabsadd L
.88571 .75679 -1.875 0 Mabsadd L closepath F
.88667 .75681 -1.875 0 Mabsadd m
.88667 .75681 0 2.5 Mabsadd L
.88667 .75681 1.875 0 Mabsadd L
.88667 .75681 0 -2.5 Mabsadd L
.88667 .75681 -1.875 0 Mabsadd L closepath F
.88762 .75682 -1.875 0 Mabsadd m
.88762 .75682 0 2.5 Mabsadd L
.88762 .75682 1.875 0 Mabsadd L
.88762 .75682 0 -2.5 Mabsadd L
.88762 .75682 -1.875 0 Mabsadd L closepath F
.88857 .75682 -1.875 0 Mabsadd m
.88857 .75682 0 2.5 Mabsadd L
.88857 .75682 1.875 0 Mabsadd L
.88857 .75682 0 -2.5 Mabsadd L
.88857 .75682 -1.875 0 Mabsadd L closepath F
.88952 .75682 -1.875 0 Mabsadd m
.88952 .75682 0 2.5 Mabsadd L
.88952 .75682 1.875 0 Mabsadd L
.88952 .75682 0 -2.5 Mabsadd L
.88952 .75682 -1.875 0 Mabsadd L closepath F
.89048 .75682 -1.875 0 Mabsadd m
.89048 .75682 0 2.5 Mabsadd L
.89048 .75682 1.875 0 Mabsadd L
.89048 .75682 0 -2.5 Mabsadd L
.89048 .75682 -1.875 0 Mabsadd L closepath F
.89143 .75682 -1.875 0 Mabsadd m
.89143 .75682 0 2.5 Mabsadd L
.89143 .75682 1.875 0 Mabsadd L
.89143 .75682 0 -2.5 Mabsadd L
.89143 .75682 -1.875 0 Mabsadd L closepath F
.89238 .75682 -1.875 0 Mabsadd m
.89238 .75682 0 2.5 Mabsadd L
.89238 .75682 1.875 0 Mabsadd L
.89238 .75682 0 -2.5 Mabsadd L
.89238 .75682 -1.875 0 Mabsadd L closepath F
.89333 .75682 -1.875 0 Mabsadd m
.89333 .75682 0 2.5 Mabsadd L
.89333 .75682 1.875 0 Mabsadd L
.89333 .75682 0 -2.5 Mabsadd L
.89333 .75682 -1.875 0 Mabsadd L closepath F
.89429 .75682 -1.875 0 Mabsadd m
.89429 .75682 0 2.5 Mabsadd L
.89429 .75682 1.875 0 Mabsadd L
.89429 .75682 0 -2.5 Mabsadd L
.89429 .75682 -1.875 0 Mabsadd L closepath F
.89524 .75682 -1.875 0 Mabsadd m
.89524 .75682 0 2.5 Mabsadd L
.89524 .75682 1.875 0 Mabsadd L
.89524 .75682 0 -2.5 Mabsadd L
.89524 .75682 -1.875 0 Mabsadd L closepath F
.89619 .75682 -1.875 0 Mabsadd m
.89619 .75682 0 2.5 Mabsadd L
.89619 .75682 1.875 0 Mabsadd L
.89619 .75682 0 -2.5 Mabsadd L
.89619 .75682 -1.875 0 Mabsadd L closepath F
.89714 .75682 -1.875 0 Mabsadd m
.89714 .75682 0 2.5 Mabsadd L
.89714 .75682 1.875 0 Mabsadd L
.89714 .75682 0 -2.5 Mabsadd L
.89714 .75682 -1.875 0 Mabsadd L closepath F
.8981 .75682 -1.875 0 Mabsadd m
.8981 .75682 0 2.5 Mabsadd L
.8981 .75682 1.875 0 Mabsadd L
.8981 .75682 0 -2.5 Mabsadd L
.8981 .75682 -1.875 0 Mabsadd L closepath F
.89905 .75682 -1.875 0 Mabsadd m
.89905 .75682 0 2.5 Mabsadd L
.89905 .75682 1.875 0 Mabsadd L
.89905 .75682 0 -2.5 Mabsadd L
.89905 .75682 -1.875 0 Mabsadd L closepath F
.9 .75682 -1.875 0 Mabsadd m
.9 .75682 0 2.5 Mabsadd L
.9 .75682 1.875 0 Mabsadd L
.9 .75682 0 -2.5 Mabsadd L
.9 .75682 -1.875 0 Mabsadd L closepath F
.90095 .75682 -1.875 0 Mabsadd m
.90095 .75682 0 2.5 Mabsadd L
.90095 .75682 1.875 0 Mabsadd L
.90095 .75682 0 -2.5 Mabsadd L
.90095 .75682 -1.875 0 Mabsadd L closepath F
.9019 .75682 -1.875 0 Mabsadd m
.9019 .75682 0 2.5 Mabsadd L
.9019 .75682 1.875 0 Mabsadd L
.9019 .75682 0 -2.5 Mabsadd L
.9019 .75682 -1.875 0 Mabsadd L closepath F
.90286 .75682 -1.875 0 Mabsadd m
.90286 .75682 0 2.5 Mabsadd L
.90286 .75682 1.875 0 Mabsadd L
.90286 .75682 0 -2.5 Mabsadd L
.90286 .75682 -1.875 0 Mabsadd L closepath F
.90381 .75682 -1.875 0 Mabsadd m
.90381 .75682 0 2.5 Mabsadd L
.90381 .75682 1.875 0 Mabsadd L
.90381 .75682 0 -2.5 Mabsadd L
.90381 .75682 -1.875 0 Mabsadd L closepath F
.90476 .75682 -1.875 0 Mabsadd m
.90476 .75682 0 2.5 Mabsadd L
.90476 .75682 1.875 0 Mabsadd L
.90476 .75682 0 -2.5 Mabsadd L
.90476 .75682 -1.875 0 Mabsadd L closepath F
.90571 .75682 -1.875 0 Mabsadd m
.90571 .75682 0 2.5 Mabsadd L
.90571 .75682 1.875 0 Mabsadd L
.90571 .75682 0 -2.5 Mabsadd L
.90571 .75682 -1.875 0 Mabsadd L closepath F
.90667 .75682 -1.875 0 Mabsadd m
.90667 .75682 0 2.5 Mabsadd L
.90667 .75682 1.875 0 Mabsadd L
.90667 .75682 0 -2.5 Mabsadd L
.90667 .75682 -1.875 0 Mabsadd L closepath F
.90762 .75682 -1.875 0 Mabsadd m
.90762 .75682 0 2.5 Mabsadd L
.90762 .75682 1.875 0 Mabsadd L
.90762 .75682 0 -2.5 Mabsadd L
.90762 .75682 -1.875 0 Mabsadd L closepath F
.90857 .75682 -1.875 0 Mabsadd m
.90857 .75682 0 2.5 Mabsadd L
.90857 .75682 1.875 0 Mabsadd L
.90857 .75682 0 -2.5 Mabsadd L
.90857 .75682 -1.875 0 Mabsadd L closepath F
.90952 .75682 -1.875 0 Mabsadd m
.90952 .75682 0 2.5 Mabsadd L
.90952 .75682 1.875 0 Mabsadd L
.90952 .75682 0 -2.5 Mabsadd L
.90952 .75682 -1.875 0 Mabsadd L closepath F
.91048 .75682 -1.875 0 Mabsadd m
.91048 .75682 0 2.5 Mabsadd L
.91048 .75682 1.875 0 Mabsadd L
.91048 .75682 0 -2.5 Mabsadd L
.91048 .75682 -1.875 0 Mabsadd L closepath F
.91143 .75682 -1.875 0 Mabsadd m
.91143 .75682 0 2.5 Mabsadd L
.91143 .75682 1.875 0 Mabsadd L
.91143 .75682 0 -2.5 Mabsadd L
.91143 .75682 -1.875 0 Mabsadd L closepath F
.91238 .75682 -1.875 0 Mabsadd m
.91238 .75682 0 2.5 Mabsadd L
.91238 .75682 1.875 0 Mabsadd L
.91238 .75682 0 -2.5 Mabsadd L
.91238 .75682 -1.875 0 Mabsadd L closepath F
.91333 .75682 -1.875 0 Mabsadd m
.91333 .75682 0 2.5 Mabsadd L
.91333 .75682 1.875 0 Mabsadd L
.91333 .75682 0 -2.5 Mabsadd L
.91333 .75682 -1.875 0 Mabsadd L closepath F
.91429 .75682 -1.875 0 Mabsadd m
.91429 .75682 0 2.5 Mabsadd L
.91429 .75682 1.875 0 Mabsadd L
.91429 .75682 0 -2.5 Mabsadd L
.91429 .75682 -1.875 0 Mabsadd L closepath F
.91524 .75682 -1.875 0 Mabsadd m
.91524 .75682 0 2.5 Mabsadd L
.91524 .75682 1.875 0 Mabsadd L
.91524 .75682 0 -2.5 Mabsadd L
.91524 .75682 -1.875 0 Mabsadd L closepath F
.91619 .75682 -1.875 0 Mabsadd m
.91619 .75682 0 2.5 Mabsadd L
.91619 .75682 1.875 0 Mabsadd L
.91619 .75682 0 -2.5 Mabsadd L
.91619 .75682 -1.875 0 Mabsadd L closepath F
.91714 .75682 -1.875 0 Mabsadd m
.91714 .75682 0 2.5 Mabsadd L
.91714 .75682 1.875 0 Mabsadd L
.91714 .75682 0 -2.5 Mabsadd L
.91714 .75682 -1.875 0 Mabsadd L closepath F
.9181 .75683 -1.875 0 Mabsadd m
.9181 .75683 0 2.5 Mabsadd L
.9181 .75683 1.875 0 Mabsadd L
.9181 .75683 0 -2.5 Mabsadd L
.9181 .75683 -1.875 0 Mabsadd L closepath F
.91905 .75683 -1.875 0 Mabsadd m
.91905 .75683 0 2.5 Mabsadd L
.91905 .75683 1.875 0 Mabsadd L
.91905 .75683 0 -2.5 Mabsadd L
.91905 .75683 -1.875 0 Mabsadd L closepath F
.92 .75683 -1.875 0 Mabsadd m
.92 .75683 0 2.5 Mabsadd L
.92 .75683 1.875 0 Mabsadd L
.92 .75683 0 -2.5 Mabsadd L
.92 .75683 -1.875 0 Mabsadd L closepath F
.92095 .75683 -1.875 0 Mabsadd m
.92095 .75683 0 2.5 Mabsadd L
.92095 .75683 1.875 0 Mabsadd L
.92095 .75683 0 -2.5 Mabsadd L
.92095 .75683 -1.875 0 Mabsadd L closepath F
.9219 .75683 -1.875 0 Mabsadd m
.9219 .75683 0 2.5 Mabsadd L
.9219 .75683 1.875 0 Mabsadd L
.9219 .75683 0 -2.5 Mabsadd L
.9219 .75683 -1.875 0 Mabsadd L closepath F
.92286 .75683 -1.875 0 Mabsadd m
.92286 .75683 0 2.5 Mabsadd L
.92286 .75683 1.875 0 Mabsadd L
.92286 .75683 0 -2.5 Mabsadd L
.92286 .75683 -1.875 0 Mabsadd L closepath F
.92381 .75682 -1.875 0 Mabsadd m
.92381 .75682 0 2.5 Mabsadd L
.92381 .75682 1.875 0 Mabsadd L
.92381 .75682 0 -2.5 Mabsadd L
.92381 .75682 -1.875 0 Mabsadd L closepath F
.92476 .75682 -1.875 0 Mabsadd m
.92476 .75682 0 2.5 Mabsadd L
.92476 .75682 1.875 0 Mabsadd L
.92476 .75682 0 -2.5 Mabsadd L
.92476 .75682 -1.875 0 Mabsadd L closepath F
.92571 .7568 -1.875 0 Mabsadd m
.92571 .7568 0 2.5 Mabsadd L
.92571 .7568 1.875 0 Mabsadd L
.92571 .7568 0 -2.5 Mabsadd L
.92571 .7568 -1.875 0 Mabsadd L closepath F
.92667 .75675 -1.875 0 Mabsadd m
.92667 .75675 0 2.5 Mabsadd L
.92667 .75675 1.875 0 Mabsadd L
.92667 .75675 0 -2.5 Mabsadd L
.92667 .75675 -1.875 0 Mabsadd L closepath F
.92762 .75664 -1.875 0 Mabsadd m
.92762 .75664 0 2.5 Mabsadd L
.92762 .75664 1.875 0 Mabsadd L
.92762 .75664 0 -2.5 Mabsadd L
.92762 .75664 -1.875 0 Mabsadd L closepath F
.92857 .75645 -1.875 0 Mabsadd m
.92857 .75645 0 2.5 Mabsadd L
.92857 .75645 1.875 0 Mabsadd L
.92857 .75645 0 -2.5 Mabsadd L
.92857 .75645 -1.875 0 Mabsadd L closepath F
.92952 .754 -1.875 0 Mabsadd m
.92952 .754 0 2.5 Mabsadd L
.92952 .754 1.875 0 Mabsadd L
.92952 .754 0 -2.5 Mabsadd L
.92952 .754 -1.875 0 Mabsadd L closepath F
.93048 .70526 -1.875 0 Mabsadd m
.93048 .70526 0 2.5 Mabsadd L
.93048 .70526 1.875 0 Mabsadd L
.93048 .70526 0 -2.5 Mabsadd L
.93048 .70526 -1.875 0 Mabsadd L closepath F
.93143 .66354 -1.875 0 Mabsadd m
.93143 .66354 0 2.5 Mabsadd L
.93143 .66354 1.875 0 Mabsadd L
.93143 .66354 0 -2.5 Mabsadd L
.93143 .66354 -1.875 0 Mabsadd L closepath F
.93238 .22641 -1.875 0 Mabsadd m
.93238 .22641 0 2.5 Mabsadd L
.93238 .22641 1.875 0 Mabsadd L
.93238 .22641 0 -2.5 Mabsadd L
.93238 .22641 -1.875 0 Mabsadd L closepath F
.93333 .82485 -1.875 0 Mabsadd m
.93333 .82485 0 2.5 Mabsadd L
.93333 .82485 1.875 0 Mabsadd L
.93333 .82485 0 -2.5 Mabsadd L
.93333 .82485 -1.875 0 Mabsadd L closepath F
.93429 .75471 -1.875 0 Mabsadd m
.93429 .75471 0 2.5 Mabsadd L
.93429 .75471 1.875 0 Mabsadd L
.93429 .75471 0 -2.5 Mabsadd L
.93429 .75471 -1.875 0 Mabsadd L closepath F
.93524 .84665 -1.875 0 Mabsadd m
.93524 .84665 0 2.5 Mabsadd L
.93524 .84665 1.875 0 Mabsadd L
.93524 .84665 0 -2.5 Mabsadd L
.93524 .84665 -1.875 0 Mabsadd L closepath F
.93619 .85657 -1.875 0 Mabsadd m
.93619 .85657 0 2.5 Mabsadd L
.93619 .85657 1.875 0 Mabsadd L
.93619 .85657 0 -2.5 Mabsadd L
.93619 .85657 -1.875 0 Mabsadd L closepath F
.93714 .85687 -1.875 0 Mabsadd m
.93714 .85687 0 2.5 Mabsadd L
.93714 .85687 1.875 0 Mabsadd L
.93714 .85687 0 -2.5 Mabsadd L
.93714 .85687 -1.875 0 Mabsadd L closepath F
.9381 .85702 -1.875 0 Mabsadd m
.9381 .85702 0 2.5 Mabsadd L
.9381 .85702 1.875 0 Mabsadd L
.9381 .85702 0 -2.5 Mabsadd L
.9381 .85702 -1.875 0 Mabsadd L closepath F
.93905 .85709 -1.875 0 Mabsadd m
.93905 .85709 0 2.5 Mabsadd L
.93905 .85709 1.875 0 Mabsadd L
.93905 .85709 0 -2.5 Mabsadd L
.93905 .85709 -1.875 0 Mabsadd L closepath F
.94 .85711 -1.875 0 Mabsadd m
.94 .85711 0 2.5 Mabsadd L
.94 .85711 1.875 0 Mabsadd L
.94 .85711 0 -2.5 Mabsadd L
.94 .85711 -1.875 0 Mabsadd L closepath F
.94095 .85712 -1.875 0 Mabsadd m
.94095 .85712 0 2.5 Mabsadd L
.94095 .85712 1.875 0 Mabsadd L
.94095 .85712 0 -2.5 Mabsadd L
.94095 .85712 -1.875 0 Mabsadd L closepath F
.9419 .85713 -1.875 0 Mabsadd m
.9419 .85713 0 2.5 Mabsadd L
.9419 .85713 1.875 0 Mabsadd L
.9419 .85713 0 -2.5 Mabsadd L
.9419 .85713 -1.875 0 Mabsadd L closepath F
.94286 .85713 -1.875 0 Mabsadd m
.94286 .85713 0 2.5 Mabsadd L
.94286 .85713 1.875 0 Mabsadd L
.94286 .85713 0 -2.5 Mabsadd L
.94286 .85713 -1.875 0 Mabsadd L closepath F
.94381 .85713 -1.875 0 Mabsadd m
.94381 .85713 0 2.5 Mabsadd L
.94381 .85713 1.875 0 Mabsadd L
.94381 .85713 0 -2.5 Mabsadd L
.94381 .85713 -1.875 0 Mabsadd L closepath F
.94476 .85713 -1.875 0 Mabsadd m
.94476 .85713 0 2.5 Mabsadd L
.94476 .85713 1.875 0 Mabsadd L
.94476 .85713 0 -2.5 Mabsadd L
.94476 .85713 -1.875 0 Mabsadd L closepath F
.94571 .85713 -1.875 0 Mabsadd m
.94571 .85713 0 2.5 Mabsadd L
.94571 .85713 1.875 0 Mabsadd L
.94571 .85713 0 -2.5 Mabsadd L
.94571 .85713 -1.875 0 Mabsadd L closepath F
.94667 .85714 -1.875 0 Mabsadd m
.94667 .85714 0 2.5 Mabsadd L
.94667 .85714 1.875 0 Mabsadd L
.94667 .85714 0 -2.5 Mabsadd L
.94667 .85714 -1.875 0 Mabsadd L closepath F
.94762 .85714 -1.875 0 Mabsadd m
.94762 .85714 0 2.5 Mabsadd L
.94762 .85714 1.875 0 Mabsadd L
.94762 .85714 0 -2.5 Mabsadd L
.94762 .85714 -1.875 0 Mabsadd L closepath F
.94857 .85713 -1.875 0 Mabsadd m
.94857 .85713 0 2.5 Mabsadd L
.94857 .85713 1.875 0 Mabsadd L
.94857 .85713 0 -2.5 Mabsadd L
.94857 .85713 -1.875 0 Mabsadd L closepath F
.94952 .85713 -1.875 0 Mabsadd m
.94952 .85713 0 2.5 Mabsadd L
.94952 .85713 1.875 0 Mabsadd L
.94952 .85713 0 -2.5 Mabsadd L
.94952 .85713 -1.875 0 Mabsadd L closepath F
.95048 .85713 -1.875 0 Mabsadd m
.95048 .85713 0 2.5 Mabsadd L
.95048 .85713 1.875 0 Mabsadd L
.95048 .85713 0 -2.5 Mabsadd L
.95048 .85713 -1.875 0 Mabsadd L closepath F
.95143 .85713 -1.875 0 Mabsadd m
.95143 .85713 0 2.5 Mabsadd L
.95143 .85713 1.875 0 Mabsadd L
.95143 .85713 0 -2.5 Mabsadd L
.95143 .85713 -1.875 0 Mabsadd L closepath F
.95238 .85713 -1.875 0 Mabsadd m
.95238 .85713 0 2.5 Mabsadd L
.95238 .85713 1.875 0 Mabsadd L
.95238 .85713 0 -2.5 Mabsadd L
.95238 .85713 -1.875 0 Mabsadd L closepath F
.95333 .85713 -1.875 0 Mabsadd m
.95333 .85713 0 2.5 Mabsadd L
.95333 .85713 1.875 0 Mabsadd L
.95333 .85713 0 -2.5 Mabsadd L
.95333 .85713 -1.875 0 Mabsadd L closepath F
.95429 .85713 -1.875 0 Mabsadd m
.95429 .85713 0 2.5 Mabsadd L
.95429 .85713 1.875 0 Mabsadd L
.95429 .85713 0 -2.5 Mabsadd L
.95429 .85713 -1.875 0 Mabsadd L closepath F
.95524 .85713 -1.875 0 Mabsadd m
.95524 .85713 0 2.5 Mabsadd L
.95524 .85713 1.875 0 Mabsadd L
.95524 .85713 0 -2.5 Mabsadd L
.95524 .85713 -1.875 0 Mabsadd L closepath F
.95619 .85713 -1.875 0 Mabsadd m
.95619 .85713 0 2.5 Mabsadd L
.95619 .85713 1.875 0 Mabsadd L
.95619 .85713 0 -2.5 Mabsadd L
.95619 .85713 -1.875 0 Mabsadd L closepath F
.95714 .85714 -1.875 0 Mabsadd m
.95714 .85714 0 2.5 Mabsadd L
.95714 .85714 1.875 0 Mabsadd L
.95714 .85714 0 -2.5 Mabsadd L
.95714 .85714 -1.875 0 Mabsadd L closepath F
.9581 .85714 -1.875 0 Mabsadd m
.9581 .85714 0 2.5 Mabsadd L
.9581 .85714 1.875 0 Mabsadd L
.9581 .85714 0 -2.5 Mabsadd L
.9581 .85714 -1.875 0 Mabsadd L closepath F
.95905 .85713 -1.875 0 Mabsadd m
.95905 .85713 0 2.5 Mabsadd L
.95905 .85713 1.875 0 Mabsadd L
.95905 .85713 0 -2.5 Mabsadd L
.95905 .85713 -1.875 0 Mabsadd L closepath F
.96 .85713 -1.875 0 Mabsadd m
.96 .85713 0 2.5 Mabsadd L
.96 .85713 1.875 0 Mabsadd L
.96 .85713 0 -2.5 Mabsadd L
.96 .85713 -1.875 0 Mabsadd L closepath F
.96095 .85714 -1.875 0 Mabsadd m
.96095 .85714 0 2.5 Mabsadd L
.96095 .85714 1.875 0 Mabsadd L
.96095 .85714 0 -2.5 Mabsadd L
.96095 .85714 -1.875 0 Mabsadd L closepath F
.9619 .85714 -1.875 0 Mabsadd m
.9619 .85714 0 2.5 Mabsadd L
.9619 .85714 1.875 0 Mabsadd L
.9619 .85714 0 -2.5 Mabsadd L
.9619 .85714 -1.875 0 Mabsadd L closepath F
.96286 .85713 -1.875 0 Mabsadd m
.96286 .85713 0 2.5 Mabsadd L
.96286 .85713 1.875 0 Mabsadd L
.96286 .85713 0 -2.5 Mabsadd L
.96286 .85713 -1.875 0 Mabsadd L closepath F
.96381 .85714 -1.875 0 Mabsadd m
.96381 .85714 0 2.5 Mabsadd L
.96381 .85714 1.875 0 Mabsadd L
.96381 .85714 0 -2.5 Mabsadd L
.96381 .85714 -1.875 0 Mabsadd L closepath F
.96476 .85713 -1.875 0 Mabsadd m
.96476 .85713 0 2.5 Mabsadd L
.96476 .85713 1.875 0 Mabsadd L
.96476 .85713 0 -2.5 Mabsadd L
.96476 .85713 -1.875 0 Mabsadd L closepath F
.96571 .85713 -1.875 0 Mabsadd m
.96571 .85713 0 2.5 Mabsadd L
.96571 .85713 1.875 0 Mabsadd L
.96571 .85713 0 -2.5 Mabsadd L
.96571 .85713 -1.875 0 Mabsadd L closepath F
.96667 .85713 -1.875 0 Mabsadd m
.96667 .85713 0 2.5 Mabsadd L
.96667 .85713 1.875 0 Mabsadd L
.96667 .85713 0 -2.5 Mabsadd L
.96667 .85713 -1.875 0 Mabsadd L closepath F
.96762 .85713 -1.875 0 Mabsadd m
.96762 .85713 0 2.5 Mabsadd L
.96762 .85713 1.875 0 Mabsadd L
.96762 .85713 0 -2.5 Mabsadd L
.96762 .85713 -1.875 0 Mabsadd L closepath F
.96857 .85713 -1.875 0 Mabsadd m
.96857 .85713 0 2.5 Mabsadd L
.96857 .85713 1.875 0 Mabsadd L
.96857 .85713 0 -2.5 Mabsadd L
.96857 .85713 -1.875 0 Mabsadd L closepath F
.96952 .85713 -1.875 0 Mabsadd m
.96952 .85713 0 2.5 Mabsadd L
.96952 .85713 1.875 0 Mabsadd L
.96952 .85713 0 -2.5 Mabsadd L
.96952 .85713 -1.875 0 Mabsadd L closepath F
.97048 .85713 -1.875 0 Mabsadd m
.97048 .85713 0 2.5 Mabsadd L
.97048 .85713 1.875 0 Mabsadd L
.97048 .85713 0 -2.5 Mabsadd L
.97048 .85713 -1.875 0 Mabsadd L closepath F
.97143 .85714 -1.875 0 Mabsadd m
.97143 .85714 0 2.5 Mabsadd L
.97143 .85714 1.875 0 Mabsadd L
.97143 .85714 0 -2.5 Mabsadd L
.97143 .85714 -1.875 0 Mabsadd L closepath F
.97238 .85714 -1.875 0 Mabsadd m
.97238 .85714 0 2.5 Mabsadd L
.97238 .85714 1.875 0 Mabsadd L
.97238 .85714 0 -2.5 Mabsadd L
.97238 .85714 -1.875 0 Mabsadd L closepath F
.97333 .85714 -1.875 0 Mabsadd m
.97333 .85714 0 2.5 Mabsadd L
.97333 .85714 1.875 0 Mabsadd L
.97333 .85714 0 -2.5 Mabsadd L
.97333 .85714 -1.875 0 Mabsadd L closepath F
.97429 .85714 -1.875 0 Mabsadd m
.97429 .85714 0 2.5 Mabsadd L
.97429 .85714 1.875 0 Mabsadd L
.97429 .85714 0 -2.5 Mabsadd L
.97429 .85714 -1.875 0 Mabsadd L closepath F
.97524 .85714 -1.875 0 Mabsadd m
.97524 .85714 0 2.5 Mabsadd L
.97524 .85714 1.875 0 Mabsadd L
.97524 .85714 0 -2.5 Mabsadd L
.97524 .85714 -1.875 0 Mabsadd L closepath F
.97619 .85714 -1.875 0 Mabsadd m
.97619 .85714 0 2.5 Mabsadd L
.97619 .85714 1.875 0 Mabsadd L
.97619 .85714 0 -2.5 Mabsadd L
.97619 .85714 -1.875 0 Mabsadd L closepath F
0 0 1 r
.02381 .02381 0 2.5 Mabsadd m
.02381 .02381 -0.56518 .7779 Mabsadd L
.02381 .02381 -2.37764 .77254 Mabsadd L
.02381 .02381 -0.91448 -0.29713 Mabsadd L
.02381 .02381 -1.46946 -2.02254 Mabsadd L
.02381 .02381 0 -0.96154 Mabsadd L
.02381 .02381 1.46946 -2.02254 Mabsadd L
.02381 .02381 .91448 -0.29713 Mabsadd L
.02381 .02381 2.37764 .77254 Mabsadd L
.02381 .02381 .56518 .7779 Mabsadd L
.02381 .02381 0 2.5 Mabsadd L closepath F
.02476 .02381 0 2.5 Mabsadd m
.02476 .02381 -0.56518 .7779 Mabsadd L
.02476 .02381 -2.37764 .77254 Mabsadd L
.02476 .02381 -0.91448 -0.29713 Mabsadd L
.02476 .02381 -1.46946 -2.02254 Mabsadd L
.02476 .02381 0 -0.96154 Mabsadd L
.02476 .02381 1.46946 -2.02254 Mabsadd L
.02476 .02381 .91448 -0.29713 Mabsadd L
.02476 .02381 2.37764 .77254 Mabsadd L
.02476 .02381 .56518 .7779 Mabsadd L
.02476 .02381 0 2.5 Mabsadd L closepath F
.02571 .02381 0 2.5 Mabsadd m
.02571 .02381 -0.56518 .7779 Mabsadd L
.02571 .02381 -2.37764 .77254 Mabsadd L
.02571 .02381 -0.91448 -0.29713 Mabsadd L
.02571 .02381 -1.46946 -2.02254 Mabsadd L
.02571 .02381 0 -0.96154 Mabsadd L
.02571 .02381 1.46946 -2.02254 Mabsadd L
.02571 .02381 .91448 -0.29713 Mabsadd L
.02571 .02381 2.37764 .77254 Mabsadd L
.02571 .02381 .56518 .7779 Mabsadd L
.02571 .02381 0 2.5 Mabsadd L closepath F
.02667 .02381 0 2.5 Mabsadd m
.02667 .02381 -0.56518 .7779 Mabsadd L
.02667 .02381 -2.37764 .77254 Mabsadd L
.02667 .02381 -0.91448 -0.29713 Mabsadd L
.02667 .02381 -1.46946 -2.02254 Mabsadd L
.02667 .02381 0 -0.96154 Mabsadd L
.02667 .02381 1.46946 -2.02254 Mabsadd L
.02667 .02381 .91448 -0.29713 Mabsadd L
.02667 .02381 2.37764 .77254 Mabsadd L
.02667 .02381 .56518 .7779 Mabsadd L
.02667 .02381 0 2.5 Mabsadd L closepath F
.02762 .02381 0 2.5 Mabsadd m
.02762 .02381 -0.56518 .7779 Mabsadd L
.02762 .02381 -2.37764 .77254 Mabsadd L
.02762 .02381 -0.91448 -0.29713 Mabsadd L
.02762 .02381 -1.46946 -2.02254 Mabsadd L
.02762 .02381 0 -0.96154 Mabsadd L
.02762 .02381 1.46946 -2.02254 Mabsadd L
.02762 .02381 .91448 -0.29713 Mabsadd L
.02762 .02381 2.37764 .77254 Mabsadd L
.02762 .02381 .56518 .7779 Mabsadd L
.02762 .02381 0 2.5 Mabsadd L closepath F
.02857 .02381 0 2.5 Mabsadd m
.02857 .02381 -0.56518 .7779 Mabsadd L
.02857 .02381 -2.37764 .77254 Mabsadd L
.02857 .02381 -0.91448 -0.29713 Mabsadd L
.02857 .02381 -1.46946 -2.02254 Mabsadd L
.02857 .02381 0 -0.96154 Mabsadd L
.02857 .02381 1.46946 -2.02254 Mabsadd L
.02857 .02381 .91448 -0.29713 Mabsadd L
.02857 .02381 2.37764 .77254 Mabsadd L
.02857 .02381 .56518 .7779 Mabsadd L
.02857 .02381 0 2.5 Mabsadd L closepath F
.02952 .02381 0 2.5 Mabsadd m
.02952 .02381 -0.56518 .7779 Mabsadd L
.02952 .02381 -2.37764 .77254 Mabsadd L
.02952 .02381 -0.91448 -0.29713 Mabsadd L
.02952 .02381 -1.46946 -2.02254 Mabsadd L
.02952 .02381 0 -0.96154 Mabsadd L
.02952 .02381 1.46946 -2.02254 Mabsadd L
.02952 .02381 .91448 -0.29713 Mabsadd L
.02952 .02381 2.37764 .77254 Mabsadd L
.02952 .02381 .56518 .7779 Mabsadd L
.02952 .02381 0 2.5 Mabsadd L closepath F
.03048 .02381 0 2.5 Mabsadd m
.03048 .02381 -0.56518 .7779 Mabsadd L
.03048 .02381 -2.37764 .77254 Mabsadd L
.03048 .02381 -0.91448 -0.29713 Mabsadd L
.03048 .02381 -1.46946 -2.02254 Mabsadd L
.03048 .02381 0 -0.96154 Mabsadd L
.03048 .02381 1.46946 -2.02254 Mabsadd L
.03048 .02381 .91448 -0.29713 Mabsadd L
.03048 .02381 2.37764 .77254 Mabsadd L
.03048 .02381 .56518 .7779 Mabsadd L
.03048 .02381 0 2.5 Mabsadd L closepath F
.03143 .02381 0 2.5 Mabsadd m
.03143 .02381 -0.56518 .7779 Mabsadd L
.03143 .02381 -2.37764 .77254 Mabsadd L
.03143 .02381 -0.91448 -0.29713 Mabsadd L
.03143 .02381 -1.46946 -2.02254 Mabsadd L
.03143 .02381 0 -0.96154 Mabsadd L
.03143 .02381 1.46946 -2.02254 Mabsadd L
.03143 .02381 .91448 -0.29713 Mabsadd L
.03143 .02381 2.37764 .77254 Mabsadd L
.03143 .02381 .56518 .7779 Mabsadd L
.03143 .02381 0 2.5 Mabsadd L closepath F
.03238 .02381 0 2.5 Mabsadd m
.03238 .02381 -0.56518 .7779 Mabsadd L
.03238 .02381 -2.37764 .77254 Mabsadd L
.03238 .02381 -0.91448 -0.29713 Mabsadd L
.03238 .02381 -1.46946 -2.02254 Mabsadd L
.03238 .02381 0 -0.96154 Mabsadd L
.03238 .02381 1.46946 -2.02254 Mabsadd L
.03238 .02381 .91448 -0.29713 Mabsadd L
.03238 .02381 2.37764 .77254 Mabsadd L
.03238 .02381 .56518 .7779 Mabsadd L
.03238 .02381 0 2.5 Mabsadd L closepath F
.03333 .02381 0 2.5 Mabsadd m
.03333 .02381 -0.56518 .7779 Mabsadd L
.03333 .02381 -2.37764 .77254 Mabsadd L
.03333 .02381 -0.91448 -0.29713 Mabsadd L
.03333 .02381 -1.46946 -2.02254 Mabsadd L
.03333 .02381 0 -0.96154 Mabsadd L
.03333 .02381 1.46946 -2.02254 Mabsadd L
.03333 .02381 .91448 -0.29713 Mabsadd L
.03333 .02381 2.37764 .77254 Mabsadd L
.03333 .02381 .56518 .7779 Mabsadd L
.03333 .02381 0 2.5 Mabsadd L closepath F
.03429 .02381 0 2.5 Mabsadd m
.03429 .02381 -0.56518 .7779 Mabsadd L
.03429 .02381 -2.37764 .77254 Mabsadd L
.03429 .02381 -0.91448 -0.29713 Mabsadd L
.03429 .02381 -1.46946 -2.02254 Mabsadd L
.03429 .02381 0 -0.96154 Mabsadd L
.03429 .02381 1.46946 -2.02254 Mabsadd L
.03429 .02381 .91448 -0.29713 Mabsadd L
.03429 .02381 2.37764 .77254 Mabsadd L
.03429 .02381 .56518 .7779 Mabsadd L
.03429 .02381 0 2.5 Mabsadd L closepath F
.03524 .02381 0 2.5 Mabsadd m
.03524 .02381 -0.56518 .7779 Mabsadd L
.03524 .02381 -2.37764 .77254 Mabsadd L
.03524 .02381 -0.91448 -0.29713 Mabsadd L
.03524 .02381 -1.46946 -2.02254 Mabsadd L
.03524 .02381 0 -0.96154 Mabsadd L
.03524 .02381 1.46946 -2.02254 Mabsadd L
.03524 .02381 .91448 -0.29713 Mabsadd L
.03524 .02381 2.37764 .77254 Mabsadd L
.03524 .02381 .56518 .7779 Mabsadd L
.03524 .02381 0 2.5 Mabsadd L closepath F
.03619 .02381 0 2.5 Mabsadd m
.03619 .02381 -0.56518 .7779 Mabsadd L
.03619 .02381 -2.37764 .77254 Mabsadd L
.03619 .02381 -0.91448 -0.29713 Mabsadd L
.03619 .02381 -1.46946 -2.02254 Mabsadd L
.03619 .02381 0 -0.96154 Mabsadd L
.03619 .02381 1.46946 -2.02254 Mabsadd L
.03619 .02381 .91448 -0.29713 Mabsadd L
.03619 .02381 2.37764 .77254 Mabsadd L
.03619 .02381 .56518 .7779 Mabsadd L
.03619 .02381 0 2.5 Mabsadd L closepath F
.03714 .02381 0 2.5 Mabsadd m
.03714 .02381 -0.56518 .7779 Mabsadd L
.03714 .02381 -2.37764 .77254 Mabsadd L
.03714 .02381 -0.91448 -0.29713 Mabsadd L
.03714 .02381 -1.46946 -2.02254 Mabsadd L
.03714 .02381 0 -0.96154 Mabsadd L
.03714 .02381 1.46946 -2.02254 Mabsadd L
.03714 .02381 .91448 -0.29713 Mabsadd L
.03714 .02381 2.37764 .77254 Mabsadd L
.03714 .02381 .56518 .7779 Mabsadd L
.03714 .02381 0 2.5 Mabsadd L closepath F
.0381 .02381 0 2.5 Mabsadd m
.0381 .02381 -0.56518 .7779 Mabsadd L
.0381 .02381 -2.37764 .77254 Mabsadd L
.0381 .02381 -0.91448 -0.29713 Mabsadd L
.0381 .02381 -1.46946 -2.02254 Mabsadd L
.0381 .02381 0 -0.96154 Mabsadd L
.0381 .02381 1.46946 -2.02254 Mabsadd L
.0381 .02381 .91448 -0.29713 Mabsadd L
.0381 .02381 2.37764 .77254 Mabsadd L
.0381 .02381 .56518 .7779 Mabsadd L
.0381 .02381 0 2.5 Mabsadd L closepath F
.03905 .02381 0 2.5 Mabsadd m
.03905 .02381 -0.56518 .7779 Mabsadd L
.03905 .02381 -2.37764 .77254 Mabsadd L
.03905 .02381 -0.91448 -0.29713 Mabsadd L
.03905 .02381 -1.46946 -2.02254 Mabsadd L
.03905 .02381 0 -0.96154 Mabsadd L
.03905 .02381 1.46946 -2.02254 Mabsadd L
.03905 .02381 .91448 -0.29713 Mabsadd L
.03905 .02381 2.37764 .77254 Mabsadd L
.03905 .02381 .56518 .7779 Mabsadd L
.03905 .02381 0 2.5 Mabsadd L closepath F
.04 .02381 0 2.5 Mabsadd m
.04 .02381 -0.56518 .7779 Mabsadd L
.04 .02381 -2.37764 .77254 Mabsadd L
.04 .02381 -0.91448 -0.29713 Mabsadd L
.04 .02381 -1.46946 -2.02254 Mabsadd L
.04 .02381 0 -0.96154 Mabsadd L
.04 .02381 1.46946 -2.02254 Mabsadd L
.04 .02381 .91448 -0.29713 Mabsadd L
.04 .02381 2.37764 .77254 Mabsadd L
.04 .02381 .56518 .7779 Mabsadd L
.04 .02381 0 2.5 Mabsadd L closepath F
.04095 .02381 0 2.5 Mabsadd m
.04095 .02381 -0.56518 .7779 Mabsadd L
.04095 .02381 -2.37764 .77254 Mabsadd L
.04095 .02381 -0.91448 -0.29713 Mabsadd L
.04095 .02381 -1.46946 -2.02254 Mabsadd L
.04095 .02381 0 -0.96154 Mabsadd L
.04095 .02381 1.46946 -2.02254 Mabsadd L
.04095 .02381 .91448 -0.29713 Mabsadd L
.04095 .02381 2.37764 .77254 Mabsadd L
.04095 .02381 .56518 .7779 Mabsadd L
.04095 .02381 0 2.5 Mabsadd L closepath F
.0419 .02381 0 2.5 Mabsadd m
.0419 .02381 -0.56518 .7779 Mabsadd L
.0419 .02381 -2.37764 .77254 Mabsadd L
.0419 .02381 -0.91448 -0.29713 Mabsadd L
.0419 .02381 -1.46946 -2.02254 Mabsadd L
.0419 .02381 0 -0.96154 Mabsadd L
.0419 .02381 1.46946 -2.02254 Mabsadd L
.0419 .02381 .91448 -0.29713 Mabsadd L
.0419 .02381 2.37764 .77254 Mabsadd L
.0419 .02381 .56518 .7779 Mabsadd L
.0419 .02381 0 2.5 Mabsadd L closepath F
.04286 .02381 0 2.5 Mabsadd m
.04286 .02381 -0.56518 .7779 Mabsadd L
.04286 .02381 -2.37764 .77254 Mabsadd L
.04286 .02381 -0.91448 -0.29713 Mabsadd L
.04286 .02381 -1.46946 -2.02254 Mabsadd L
.04286 .02381 0 -0.96154 Mabsadd L
.04286 .02381 1.46946 -2.02254 Mabsadd L
.04286 .02381 .91448 -0.29713 Mabsadd L
.04286 .02381 2.37764 .77254 Mabsadd L
.04286 .02381 .56518 .7779 Mabsadd L
.04286 .02381 0 2.5 Mabsadd L closepath F
.04381 .02381 0 2.5 Mabsadd m
.04381 .02381 -0.56518 .7779 Mabsadd L
.04381 .02381 -2.37764 .77254 Mabsadd L
.04381 .02381 -0.91448 -0.29713 Mabsadd L
.04381 .02381 -1.46946 -2.02254 Mabsadd L
.04381 .02381 0 -0.96154 Mabsadd L
.04381 .02381 1.46946 -2.02254 Mabsadd L
.04381 .02381 .91448 -0.29713 Mabsadd L
.04381 .02381 2.37764 .77254 Mabsadd L
.04381 .02381 .56518 .7779 Mabsadd L
.04381 .02381 0 2.5 Mabsadd L closepath F
.04476 .02381 0 2.5 Mabsadd m
.04476 .02381 -0.56518 .7779 Mabsadd L
.04476 .02381 -2.37764 .77254 Mabsadd L
.04476 .02381 -0.91448 -0.29713 Mabsadd L
.04476 .02381 -1.46946 -2.02254 Mabsadd L
.04476 .02381 0 -0.96154 Mabsadd L
.04476 .02381 1.46946 -2.02254 Mabsadd L
.04476 .02381 .91448 -0.29713 Mabsadd L
.04476 .02381 2.37764 .77254 Mabsadd L
.04476 .02381 .56518 .7779 Mabsadd L
.04476 .02381 0 2.5 Mabsadd L closepath F
.04571 .02381 0 2.5 Mabsadd m
.04571 .02381 -0.56518 .7779 Mabsadd L
.04571 .02381 -2.37764 .77254 Mabsadd L
.04571 .02381 -0.91448 -0.29713 Mabsadd L
.04571 .02381 -1.46946 -2.02254 Mabsadd L
.04571 .02381 0 -0.96154 Mabsadd L
.04571 .02381 1.46946 -2.02254 Mabsadd L
.04571 .02381 .91448 -0.29713 Mabsadd L
.04571 .02381 2.37764 .77254 Mabsadd L
.04571 .02381 .56518 .7779 Mabsadd L
.04571 .02381 0 2.5 Mabsadd L closepath F
.04667 .02381 0 2.5 Mabsadd m
.04667 .02381 -0.56518 .7779 Mabsadd L
.04667 .02381 -2.37764 .77254 Mabsadd L
.04667 .02381 -0.91448 -0.29713 Mabsadd L
.04667 .02381 -1.46946 -2.02254 Mabsadd L
.04667 .02381 0 -0.96154 Mabsadd L
.04667 .02381 1.46946 -2.02254 Mabsadd L
.04667 .02381 .91448 -0.29713 Mabsadd L
.04667 .02381 2.37764 .77254 Mabsadd L
.04667 .02381 .56518 .7779 Mabsadd L
.04667 .02381 0 2.5 Mabsadd L closepath F
.04762 .02381 0 2.5 Mabsadd m
.04762 .02381 -0.56518 .7779 Mabsadd L
.04762 .02381 -2.37764 .77254 Mabsadd L
.04762 .02381 -0.91448 -0.29713 Mabsadd L
.04762 .02381 -1.46946 -2.02254 Mabsadd L
.04762 .02381 0 -0.96154 Mabsadd L
.04762 .02381 1.46946 -2.02254 Mabsadd L
.04762 .02381 .91448 -0.29713 Mabsadd L
.04762 .02381 2.37764 .77254 Mabsadd L
.04762 .02381 .56518 .7779 Mabsadd L
.04762 .02381 0 2.5 Mabsadd L closepath F
.04857 .02381 0 2.5 Mabsadd m
.04857 .02381 -0.56518 .7779 Mabsadd L
.04857 .02381 -2.37764 .77254 Mabsadd L
.04857 .02381 -0.91448 -0.29713 Mabsadd L
.04857 .02381 -1.46946 -2.02254 Mabsadd L
.04857 .02381 0 -0.96154 Mabsadd L
.04857 .02381 1.46946 -2.02254 Mabsadd L
.04857 .02381 .91448 -0.29713 Mabsadd L
.04857 .02381 2.37764 .77254 Mabsadd L
.04857 .02381 .56518 .7779 Mabsadd L
.04857 .02381 0 2.5 Mabsadd L closepath F
.04952 .02381 0 2.5 Mabsadd m
.04952 .02381 -0.56518 .7779 Mabsadd L
.04952 .02381 -2.37764 .77254 Mabsadd L
.04952 .02381 -0.91448 -0.29713 Mabsadd L
.04952 .02381 -1.46946 -2.02254 Mabsadd L
.04952 .02381 0 -0.96154 Mabsadd L
.04952 .02381 1.46946 -2.02254 Mabsadd L
.04952 .02381 .91448 -0.29713 Mabsadd L
.04952 .02381 2.37764 .77254 Mabsadd L
.04952 .02381 .56518 .7779 Mabsadd L
.04952 .02381 0 2.5 Mabsadd L closepath F
.05048 .02381 0 2.5 Mabsadd m
.05048 .02381 -0.56518 .7779 Mabsadd L
.05048 .02381 -2.37764 .77254 Mabsadd L
.05048 .02381 -0.91448 -0.29713 Mabsadd L
.05048 .02381 -1.46946 -2.02254 Mabsadd L
.05048 .02381 0 -0.96154 Mabsadd L
.05048 .02381 1.46946 -2.02254 Mabsadd L
.05048 .02381 .91448 -0.29713 Mabsadd L
.05048 .02381 2.37764 .77254 Mabsadd L
.05048 .02381 .56518 .7779 Mabsadd L
.05048 .02381 0 2.5 Mabsadd L closepath F
.05143 .02381 0 2.5 Mabsadd m
.05143 .02381 -0.56518 .7779 Mabsadd L
.05143 .02381 -2.37764 .77254 Mabsadd L
.05143 .02381 -0.91448 -0.29713 Mabsadd L
.05143 .02381 -1.46946 -2.02254 Mabsadd L
.05143 .02381 0 -0.96154 Mabsadd L
.05143 .02381 1.46946 -2.02254 Mabsadd L
.05143 .02381 .91448 -0.29713 Mabsadd L
.05143 .02381 2.37764 .77254 Mabsadd L
.05143 .02381 .56518 .7779 Mabsadd L
.05143 .02381 0 2.5 Mabsadd L closepath F
.05238 .02381 0 2.5 Mabsadd m
.05238 .02381 -0.56518 .7779 Mabsadd L
.05238 .02381 -2.37764 .77254 Mabsadd L
.05238 .02381 -0.91448 -0.29713 Mabsadd L
.05238 .02381 -1.46946 -2.02254 Mabsadd L
.05238 .02381 0 -0.96154 Mabsadd L
.05238 .02381 1.46946 -2.02254 Mabsadd L
.05238 .02381 .91448 -0.29713 Mabsadd L
.05238 .02381 2.37764 .77254 Mabsadd L
.05238 .02381 .56518 .7779 Mabsadd L
.05238 .02381 0 2.5 Mabsadd L closepath F
.05333 .02381 0 2.5 Mabsadd m
.05333 .02381 -0.56518 .7779 Mabsadd L
.05333 .02381 -2.37764 .77254 Mabsadd L
.05333 .02381 -0.91448 -0.29713 Mabsadd L
.05333 .02381 -1.46946 -2.02254 Mabsadd L
.05333 .02381 0 -0.96154 Mabsadd L
.05333 .02381 1.46946 -2.02254 Mabsadd L
.05333 .02381 .91448 -0.29713 Mabsadd L
.05333 .02381 2.37764 .77254 Mabsadd L
.05333 .02381 .56518 .7779 Mabsadd L
.05333 .02381 0 2.5 Mabsadd L closepath F
.05429 .02381 0 2.5 Mabsadd m
.05429 .02381 -0.56518 .7779 Mabsadd L
.05429 .02381 -2.37764 .77254 Mabsadd L
.05429 .02381 -0.91448 -0.29713 Mabsadd L
.05429 .02381 -1.46946 -2.02254 Mabsadd L
.05429 .02381 0 -0.96154 Mabsadd L
.05429 .02381 1.46946 -2.02254 Mabsadd L
.05429 .02381 .91448 -0.29713 Mabsadd L
.05429 .02381 2.37764 .77254 Mabsadd L
.05429 .02381 .56518 .7779 Mabsadd L
.05429 .02381 0 2.5 Mabsadd L closepath F
.05524 .02381 0 2.5 Mabsadd m
.05524 .02381 -0.56518 .7779 Mabsadd L
.05524 .02381 -2.37764 .77254 Mabsadd L
.05524 .02381 -0.91448 -0.29713 Mabsadd L
.05524 .02381 -1.46946 -2.02254 Mabsadd L
.05524 .02381 0 -0.96154 Mabsadd L
.05524 .02381 1.46946 -2.02254 Mabsadd L
.05524 .02381 .91448 -0.29713 Mabsadd L
.05524 .02381 2.37764 .77254 Mabsadd L
.05524 .02381 .56518 .7779 Mabsadd L
.05524 .02381 0 2.5 Mabsadd L closepath F
.05619 .02381 0 2.5 Mabsadd m
.05619 .02381 -0.56518 .7779 Mabsadd L
.05619 .02381 -2.37764 .77254 Mabsadd L
.05619 .02381 -0.91448 -0.29713 Mabsadd L
.05619 .02381 -1.46946 -2.02254 Mabsadd L
.05619 .02381 0 -0.96154 Mabsadd L
.05619 .02381 1.46946 -2.02254 Mabsadd L
.05619 .02381 .91448 -0.29713 Mabsadd L
.05619 .02381 2.37764 .77254 Mabsadd L
.05619 .02381 .56518 .7779 Mabsadd L
.05619 .02381 0 2.5 Mabsadd L closepath F
.05714 .02381 0 2.5 Mabsadd m
.05714 .02381 -0.56518 .7779 Mabsadd L
.05714 .02381 -2.37764 .77254 Mabsadd L
.05714 .02381 -0.91448 -0.29713 Mabsadd L
.05714 .02381 -1.46946 -2.02254 Mabsadd L
.05714 .02381 0 -0.96154 Mabsadd L
.05714 .02381 1.46946 -2.02254 Mabsadd L
.05714 .02381 .91448 -0.29713 Mabsadd L
.05714 .02381 2.37764 .77254 Mabsadd L
.05714 .02381 .56518 .7779 Mabsadd L
.05714 .02381 0 2.5 Mabsadd L closepath F
.0581 .02381 0 2.5 Mabsadd m
.0581 .02381 -0.56518 .7779 Mabsadd L
.0581 .02381 -2.37764 .77254 Mabsadd L
.0581 .02381 -0.91448 -0.29713 Mabsadd L
.0581 .02381 -1.46946 -2.02254 Mabsadd L
.0581 .02381 0 -0.96154 Mabsadd L
.0581 .02381 1.46946 -2.02254 Mabsadd L
.0581 .02381 .91448 -0.29713 Mabsadd L
.0581 .02381 2.37764 .77254 Mabsadd L
.0581 .02381 .56518 .7779 Mabsadd L
.0581 .02381 0 2.5 Mabsadd L closepath F
.05905 .02381 0 2.5 Mabsadd m
.05905 .02381 -0.56518 .7779 Mabsadd L
.05905 .02381 -2.37764 .77254 Mabsadd L
.05905 .02381 -0.91448 -0.29713 Mabsadd L
.05905 .02381 -1.46946 -2.02254 Mabsadd L
.05905 .02381 0 -0.96154 Mabsadd L
.05905 .02381 1.46946 -2.02254 Mabsadd L
.05905 .02381 .91448 -0.29713 Mabsadd L
.05905 .02381 2.37764 .77254 Mabsadd L
.05905 .02381 .56518 .7779 Mabsadd L
.05905 .02381 0 2.5 Mabsadd L closepath F
.06 .02381 0 2.5 Mabsadd m
.06 .02381 -0.56518 .7779 Mabsadd L
.06 .02381 -2.37764 .77254 Mabsadd L
.06 .02381 -0.91448 -0.29713 Mabsadd L
.06 .02381 -1.46946 -2.02254 Mabsadd L
.06 .02381 0 -0.96154 Mabsadd L
.06 .02381 1.46946 -2.02254 Mabsadd L
.06 .02381 .91448 -0.29713 Mabsadd L
.06 .02381 2.37764 .77254 Mabsadd L
.06 .02381 .56518 .7779 Mabsadd L
.06 .02381 0 2.5 Mabsadd L closepath F
.06095 .02381 0 2.5 Mabsadd m
.06095 .02381 -0.56518 .7779 Mabsadd L
.06095 .02381 -2.37764 .77254 Mabsadd L
.06095 .02381 -0.91448 -0.29713 Mabsadd L
.06095 .02381 -1.46946 -2.02254 Mabsadd L
.06095 .02381 0 -0.96154 Mabsadd L
.06095 .02381 1.46946 -2.02254 Mabsadd L
.06095 .02381 .91448 -0.29713 Mabsadd L
.06095 .02381 2.37764 .77254 Mabsadd L
.06095 .02381 .56518 .7779 Mabsadd L
.06095 .02381 0 2.5 Mabsadd L closepath F
.0619 .02381 0 2.5 Mabsadd m
.0619 .02381 -0.56518 .7779 Mabsadd L
.0619 .02381 -2.37764 .77254 Mabsadd L
.0619 .02381 -0.91448 -0.29713 Mabsadd L
.0619 .02381 -1.46946 -2.02254 Mabsadd L
.0619 .02381 0 -0.96154 Mabsadd L
.0619 .02381 1.46946 -2.02254 Mabsadd L
.0619 .02381 .91448 -0.29713 Mabsadd L
.0619 .02381 2.37764 .77254 Mabsadd L
.0619 .02381 .56518 .7779 Mabsadd L
.0619 .02381 0 2.5 Mabsadd L closepath F
.06286 .02381 0 2.5 Mabsadd m
.06286 .02381 -0.56518 .7779 Mabsadd L
.06286 .02381 -2.37764 .77254 Mabsadd L
.06286 .02381 -0.91448 -0.29713 Mabsadd L
.06286 .02381 -1.46946 -2.02254 Mabsadd L
.06286 .02381 0 -0.96154 Mabsadd L
.06286 .02381 1.46946 -2.02254 Mabsadd L
.06286 .02381 .91448 -0.29713 Mabsadd L
.06286 .02381 2.37764 .77254 Mabsadd L
.06286 .02381 .56518 .7779 Mabsadd L
.06286 .02381 0 2.5 Mabsadd L closepath F
.06381 .02381 0 2.5 Mabsadd m
.06381 .02381 -0.56518 .7779 Mabsadd L
.06381 .02381 -2.37764 .77254 Mabsadd L
.06381 .02381 -0.91448 -0.29713 Mabsadd L
.06381 .02381 -1.46946 -2.02254 Mabsadd L
.06381 .02381 0 -0.96154 Mabsadd L
.06381 .02381 1.46946 -2.02254 Mabsadd L
.06381 .02381 .91448 -0.29713 Mabsadd L
.06381 .02381 2.37764 .77254 Mabsadd L
.06381 .02381 .56518 .7779 Mabsadd L
.06381 .02381 0 2.5 Mabsadd L closepath F
.06476 .02381 0 2.5 Mabsadd m
.06476 .02381 -0.56518 .7779 Mabsadd L
.06476 .02381 -2.37764 .77254 Mabsadd L
.06476 .02381 -0.91448 -0.29713 Mabsadd L
.06476 .02381 -1.46946 -2.02254 Mabsadd L
.06476 .02381 0 -0.96154 Mabsadd L
.06476 .02381 1.46946 -2.02254 Mabsadd L
.06476 .02381 .91448 -0.29713 Mabsadd L
.06476 .02381 2.37764 .77254 Mabsadd L
.06476 .02381 .56518 .7779 Mabsadd L
.06476 .02381 0 2.5 Mabsadd L closepath F
.06571 .02381 0 2.5 Mabsadd m
.06571 .02381 -0.56518 .7779 Mabsadd L
.06571 .02381 -2.37764 .77254 Mabsadd L
.06571 .02381 -0.91448 -0.29713 Mabsadd L
.06571 .02381 -1.46946 -2.02254 Mabsadd L
.06571 .02381 0 -0.96154 Mabsadd L
.06571 .02381 1.46946 -2.02254 Mabsadd L
.06571 .02381 .91448 -0.29713 Mabsadd L
.06571 .02381 2.37764 .77254 Mabsadd L
.06571 .02381 .56518 .7779 Mabsadd L
.06571 .02381 0 2.5 Mabsadd L closepath F
.06667 .02381 0 2.5 Mabsadd m
.06667 .02381 -0.56518 .7779 Mabsadd L
.06667 .02381 -2.37764 .77254 Mabsadd L
.06667 .02381 -0.91448 -0.29713 Mabsadd L
.06667 .02381 -1.46946 -2.02254 Mabsadd L
.06667 .02381 0 -0.96154 Mabsadd L
.06667 .02381 1.46946 -2.02254 Mabsadd L
.06667 .02381 .91448 -0.29713 Mabsadd L
.06667 .02381 2.37764 .77254 Mabsadd L
.06667 .02381 .56518 .7779 Mabsadd L
.06667 .02381 0 2.5 Mabsadd L closepath F
.06762 .02381 0 2.5 Mabsadd m
.06762 .02381 -0.56518 .7779 Mabsadd L
.06762 .02381 -2.37764 .77254 Mabsadd L
.06762 .02381 -0.91448 -0.29713 Mabsadd L
.06762 .02381 -1.46946 -2.02254 Mabsadd L
.06762 .02381 0 -0.96154 Mabsadd L
.06762 .02381 1.46946 -2.02254 Mabsadd L
.06762 .02381 .91448 -0.29713 Mabsadd L
.06762 .02381 2.37764 .77254 Mabsadd L
.06762 .02381 .56518 .7779 Mabsadd L
.06762 .02381 0 2.5 Mabsadd L closepath F
.06857 .02381 0 2.5 Mabsadd m
.06857 .02381 -0.56518 .7779 Mabsadd L
.06857 .02381 -2.37764 .77254 Mabsadd L
.06857 .02381 -0.91448 -0.29713 Mabsadd L
.06857 .02381 -1.46946 -2.02254 Mabsadd L
.06857 .02381 0 -0.96154 Mabsadd L
.06857 .02381 1.46946 -2.02254 Mabsadd L
.06857 .02381 .91448 -0.29713 Mabsadd L
.06857 .02381 2.37764 .77254 Mabsadd L
.06857 .02381 .56518 .7779 Mabsadd L
.06857 .02381 0 2.5 Mabsadd L closepath F
.06952 .02381 0 2.5 Mabsadd m
.06952 .02381 -0.56518 .7779 Mabsadd L
.06952 .02381 -2.37764 .77254 Mabsadd L
.06952 .02381 -0.91448 -0.29713 Mabsadd L
.06952 .02381 -1.46946 -2.02254 Mabsadd L
.06952 .02381 0 -0.96154 Mabsadd L
.06952 .02381 1.46946 -2.02254 Mabsadd L
.06952 .02381 .91448 -0.29713 Mabsadd L
.06952 .02381 2.37764 .77254 Mabsadd L
.06952 .02381 .56518 .7779 Mabsadd L
.06952 .02381 0 2.5 Mabsadd L closepath F
.07048 .02381 0 2.5 Mabsadd m
.07048 .02381 -0.56518 .7779 Mabsadd L
.07048 .02381 -2.37764 .77254 Mabsadd L
.07048 .02381 -0.91448 -0.29713 Mabsadd L
.07048 .02381 -1.46946 -2.02254 Mabsadd L
.07048 .02381 0 -0.96154 Mabsadd L
.07048 .02381 1.46946 -2.02254 Mabsadd L
.07048 .02381 .91448 -0.29713 Mabsadd L
.07048 .02381 2.37764 .77254 Mabsadd L
.07048 .02381 .56518 .7779 Mabsadd L
.07048 .02381 0 2.5 Mabsadd L closepath F
.07143 .02381 0 2.5 Mabsadd m
.07143 .02381 -0.56518 .7779 Mabsadd L
.07143 .02381 -2.37764 .77254 Mabsadd L
.07143 .02381 -0.91448 -0.29713 Mabsadd L
.07143 .02381 -1.46946 -2.02254 Mabsadd L
.07143 .02381 0 -0.96154 Mabsadd L
.07143 .02381 1.46946 -2.02254 Mabsadd L
.07143 .02381 .91448 -0.29713 Mabsadd L
.07143 .02381 2.37764 .77254 Mabsadd L
.07143 .02381 .56518 .7779 Mabsadd L
.07143 .02381 0 2.5 Mabsadd L closepath F
.07238 .02381 0 2.5 Mabsadd m
.07238 .02381 -0.56518 .7779 Mabsadd L
.07238 .02381 -2.37764 .77254 Mabsadd L
.07238 .02381 -0.91448 -0.29713 Mabsadd L
.07238 .02381 -1.46946 -2.02254 Mabsadd L
.07238 .02381 0 -0.96154 Mabsadd L
.07238 .02381 1.46946 -2.02254 Mabsadd L
.07238 .02381 .91448 -0.29713 Mabsadd L
.07238 .02381 2.37764 .77254 Mabsadd L
.07238 .02381 .56518 .7779 Mabsadd L
.07238 .02381 0 2.5 Mabsadd L closepath F
.07333 .02381 0 2.5 Mabsadd m
.07333 .02381 -0.56518 .7779 Mabsadd L
.07333 .02381 -2.37764 .77254 Mabsadd L
.07333 .02381 -0.91448 -0.29713 Mabsadd L
.07333 .02381 -1.46946 -2.02254 Mabsadd L
.07333 .02381 0 -0.96154 Mabsadd L
.07333 .02381 1.46946 -2.02254 Mabsadd L
.07333 .02381 .91448 -0.29713 Mabsadd L
.07333 .02381 2.37764 .77254 Mabsadd L
.07333 .02381 .56518 .7779 Mabsadd L
.07333 .02381 0 2.5 Mabsadd L closepath F
.07429 .02381 0 2.5 Mabsadd m
.07429 .02381 -0.56518 .7779 Mabsadd L
.07429 .02381 -2.37764 .77254 Mabsadd L
.07429 .02381 -0.91448 -0.29713 Mabsadd L
.07429 .02381 -1.46946 -2.02254 Mabsadd L
.07429 .02381 0 -0.96154 Mabsadd L
.07429 .02381 1.46946 -2.02254 Mabsadd L
.07429 .02381 .91448 -0.29713 Mabsadd L
.07429 .02381 2.37764 .77254 Mabsadd L
.07429 .02381 .56518 .7779 Mabsadd L
.07429 .02381 0 2.5 Mabsadd L closepath F
.07524 .02381 0 2.5 Mabsadd m
.07524 .02381 -0.56518 .7779 Mabsadd L
.07524 .02381 -2.37764 .77254 Mabsadd L
.07524 .02381 -0.91448 -0.29713 Mabsadd L
.07524 .02381 -1.46946 -2.02254 Mabsadd L
.07524 .02381 0 -0.96154 Mabsadd L
.07524 .02381 1.46946 -2.02254 Mabsadd L
.07524 .02381 .91448 -0.29713 Mabsadd L
.07524 .02381 2.37764 .77254 Mabsadd L
.07524 .02381 .56518 .7779 Mabsadd L
.07524 .02381 0 2.5 Mabsadd L closepath F
.07619 .02381 0 2.5 Mabsadd m
.07619 .02381 -0.56518 .7779 Mabsadd L
.07619 .02381 -2.37764 .77254 Mabsadd L
.07619 .02381 -0.91448 -0.29713 Mabsadd L
.07619 .02381 -1.46946 -2.02254 Mabsadd L
.07619 .02381 0 -0.96154 Mabsadd L
.07619 .02381 1.46946 -2.02254 Mabsadd L
.07619 .02381 .91448 -0.29713 Mabsadd L
.07619 .02381 2.37764 .77254 Mabsadd L
.07619 .02381 .56518 .7779 Mabsadd L
.07619 .02381 0 2.5 Mabsadd L closepath F
.07714 .02381 0 2.5 Mabsadd m
.07714 .02381 -0.56518 .7779 Mabsadd L
.07714 .02381 -2.37764 .77254 Mabsadd L
.07714 .02381 -0.91448 -0.29713 Mabsadd L
.07714 .02381 -1.46946 -2.02254 Mabsadd L
.07714 .02381 0 -0.96154 Mabsadd L
.07714 .02381 1.46946 -2.02254 Mabsadd L
.07714 .02381 .91448 -0.29713 Mabsadd L
.07714 .02381 2.37764 .77254 Mabsadd L
.07714 .02381 .56518 .7779 Mabsadd L
.07714 .02381 0 2.5 Mabsadd L closepath F
.0781 .02381 0 2.5 Mabsadd m
.0781 .02381 -0.56518 .7779 Mabsadd L
.0781 .02381 -2.37764 .77254 Mabsadd L
.0781 .02381 -0.91448 -0.29713 Mabsadd L
.0781 .02381 -1.46946 -2.02254 Mabsadd L
.0781 .02381 0 -0.96154 Mabsadd L
.0781 .02381 1.46946 -2.02254 Mabsadd L
.0781 .02381 .91448 -0.29713 Mabsadd L
.0781 .02381 2.37764 .77254 Mabsadd L
.0781 .02381 .56518 .7779 Mabsadd L
.0781 .02381 0 2.5 Mabsadd L closepath F
.07905 .02381 0 2.5 Mabsadd m
.07905 .02381 -0.56518 .7779 Mabsadd L
.07905 .02381 -2.37764 .77254 Mabsadd L
.07905 .02381 -0.91448 -0.29713 Mabsadd L
.07905 .02381 -1.46946 -2.02254 Mabsadd L
.07905 .02381 0 -0.96154 Mabsadd L
.07905 .02381 1.46946 -2.02254 Mabsadd L
.07905 .02381 .91448 -0.29713 Mabsadd L
.07905 .02381 2.37764 .77254 Mabsadd L
.07905 .02381 .56518 .7779 Mabsadd L
.07905 .02381 0 2.5 Mabsadd L closepath F
.08 .02381 0 2.5 Mabsadd m
.08 .02381 -0.56518 .7779 Mabsadd L
.08 .02381 -2.37764 .77254 Mabsadd L
.08 .02381 -0.91448 -0.29713 Mabsadd L
.08 .02381 -1.46946 -2.02254 Mabsadd L
.08 .02381 0 -0.96154 Mabsadd L
.08 .02381 1.46946 -2.02254 Mabsadd L
.08 .02381 .91448 -0.29713 Mabsadd L
.08 .02381 2.37764 .77254 Mabsadd L
.08 .02381 .56518 .7779 Mabsadd L
.08 .02381 0 2.5 Mabsadd L closepath F
.08095 .02381 0 2.5 Mabsadd m
.08095 .02381 -0.56518 .7779 Mabsadd L
.08095 .02381 -2.37764 .77254 Mabsadd L
.08095 .02381 -0.91448 -0.29713 Mabsadd L
.08095 .02381 -1.46946 -2.02254 Mabsadd L
.08095 .02381 0 -0.96154 Mabsadd L
.08095 .02381 1.46946 -2.02254 Mabsadd L
.08095 .02381 .91448 -0.29713 Mabsadd L
.08095 .02381 2.37764 .77254 Mabsadd L
.08095 .02381 .56518 .7779 Mabsadd L
.08095 .02381 0 2.5 Mabsadd L closepath F
.0819 .02382 0 2.5 Mabsadd m
.0819 .02382 -0.56518 .7779 Mabsadd L
.0819 .02382 -2.37764 .77254 Mabsadd L
.0819 .02382 -0.91448 -0.29713 Mabsadd L
.0819 .02382 -1.46946 -2.02254 Mabsadd L
.0819 .02382 0 -0.96154 Mabsadd L
.0819 .02382 1.46946 -2.02254 Mabsadd L
.0819 .02382 .91448 -0.29713 Mabsadd L
.0819 .02382 2.37764 .77254 Mabsadd L
.0819 .02382 .56518 .7779 Mabsadd L
.0819 .02382 0 2.5 Mabsadd L closepath F
.08286 .02382 0 2.5 Mabsadd m
.08286 .02382 -0.56518 .7779 Mabsadd L
.08286 .02382 -2.37764 .77254 Mabsadd L
.08286 .02382 -0.91448 -0.29713 Mabsadd L
.08286 .02382 -1.46946 -2.02254 Mabsadd L
.08286 .02382 0 -0.96154 Mabsadd L
.08286 .02382 1.46946 -2.02254 Mabsadd L
.08286 .02382 .91448 -0.29713 Mabsadd L
.08286 .02382 2.37764 .77254 Mabsadd L
.08286 .02382 .56518 .7779 Mabsadd L
.08286 .02382 0 2.5 Mabsadd L closepath F
.08381 .02382 0 2.5 Mabsadd m
.08381 .02382 -0.56518 .7779 Mabsadd L
.08381 .02382 -2.37764 .77254 Mabsadd L
.08381 .02382 -0.91448 -0.29713 Mabsadd L
.08381 .02382 -1.46946 -2.02254 Mabsadd L
.08381 .02382 0 -0.96154 Mabsadd L
.08381 .02382 1.46946 -2.02254 Mabsadd L
.08381 .02382 .91448 -0.29713 Mabsadd L
.08381 .02382 2.37764 .77254 Mabsadd L
.08381 .02382 .56518 .7779 Mabsadd L
.08381 .02382 0 2.5 Mabsadd L closepath F
.08476 .02381 0 2.5 Mabsadd m
.08476 .02381 -0.56518 .7779 Mabsadd L
.08476 .02381 -2.37764 .77254 Mabsadd L
.08476 .02381 -0.91448 -0.29713 Mabsadd L
.08476 .02381 -1.46946 -2.02254 Mabsadd L
.08476 .02381 0 -0.96154 Mabsadd L
.08476 .02381 1.46946 -2.02254 Mabsadd L
.08476 .02381 .91448 -0.29713 Mabsadd L
.08476 .02381 2.37764 .77254 Mabsadd L
.08476 .02381 .56518 .7779 Mabsadd L
.08476 .02381 0 2.5 Mabsadd L closepath F
.08571 .02382 0 2.5 Mabsadd m
.08571 .02382 -0.56518 .7779 Mabsadd L
.08571 .02382 -2.37764 .77254 Mabsadd L
.08571 .02382 -0.91448 -0.29713 Mabsadd L
.08571 .02382 -1.46946 -2.02254 Mabsadd L
.08571 .02382 0 -0.96154 Mabsadd L
.08571 .02382 1.46946 -2.02254 Mabsadd L
.08571 .02382 .91448 -0.29713 Mabsadd L
.08571 .02382 2.37764 .77254 Mabsadd L
.08571 .02382 .56518 .7779 Mabsadd L
.08571 .02382 0 2.5 Mabsadd L closepath F
.08667 .02381 0 2.5 Mabsadd m
.08667 .02381 -0.56518 .7779 Mabsadd L
.08667 .02381 -2.37764 .77254 Mabsadd L
.08667 .02381 -0.91448 -0.29713 Mabsadd L
.08667 .02381 -1.46946 -2.02254 Mabsadd L
.08667 .02381 0 -0.96154 Mabsadd L
.08667 .02381 1.46946 -2.02254 Mabsadd L
.08667 .02381 .91448 -0.29713 Mabsadd L
.08667 .02381 2.37764 .77254 Mabsadd L
.08667 .02381 .56518 .7779 Mabsadd L
.08667 .02381 0 2.5 Mabsadd L closepath F
.08762 .02382 0 2.5 Mabsadd m
.08762 .02382 -0.56518 .7779 Mabsadd L
.08762 .02382 -2.37764 .77254 Mabsadd L
.08762 .02382 -0.91448 -0.29713 Mabsadd L
.08762 .02382 -1.46946 -2.02254 Mabsadd L
.08762 .02382 0 -0.96154 Mabsadd L
.08762 .02382 1.46946 -2.02254 Mabsadd L
.08762 .02382 .91448 -0.29713 Mabsadd L
.08762 .02382 2.37764 .77254 Mabsadd L
.08762 .02382 .56518 .7779 Mabsadd L
.08762 .02382 0 2.5 Mabsadd L closepath F
.08857 .02382 0 2.5 Mabsadd m
.08857 .02382 -0.56518 .7779 Mabsadd L
.08857 .02382 -2.37764 .77254 Mabsadd L
.08857 .02382 -0.91448 -0.29713 Mabsadd L
.08857 .02382 -1.46946 -2.02254 Mabsadd L
.08857 .02382 0 -0.96154 Mabsadd L
.08857 .02382 1.46946 -2.02254 Mabsadd L
.08857 .02382 .91448 -0.29713 Mabsadd L
.08857 .02382 2.37764 .77254 Mabsadd L
.08857 .02382 .56518 .7779 Mabsadd L
.08857 .02382 0 2.5 Mabsadd L closepath F
.08952 .02382 0 2.5 Mabsadd m
.08952 .02382 -0.56518 .7779 Mabsadd L
.08952 .02382 -2.37764 .77254 Mabsadd L
.08952 .02382 -0.91448 -0.29713 Mabsadd L
.08952 .02382 -1.46946 -2.02254 Mabsadd L
.08952 .02382 0 -0.96154 Mabsadd L
.08952 .02382 1.46946 -2.02254 Mabsadd L
.08952 .02382 .91448 -0.29713 Mabsadd L
.08952 .02382 2.37764 .77254 Mabsadd L
.08952 .02382 .56518 .7779 Mabsadd L
.08952 .02382 0 2.5 Mabsadd L closepath F
.09048 .02382 0 2.5 Mabsadd m
.09048 .02382 -0.56518 .7779 Mabsadd L
.09048 .02382 -2.37764 .77254 Mabsadd L
.09048 .02382 -0.91448 -0.29713 Mabsadd L
.09048 .02382 -1.46946 -2.02254 Mabsadd L
.09048 .02382 0 -0.96154 Mabsadd L
.09048 .02382 1.46946 -2.02254 Mabsadd L
.09048 .02382 .91448 -0.29713 Mabsadd L
.09048 .02382 2.37764 .77254 Mabsadd L
.09048 .02382 .56518 .7779 Mabsadd L
.09048 .02382 0 2.5 Mabsadd L closepath F
.09143 .02382 0 2.5 Mabsadd m
.09143 .02382 -0.56518 .7779 Mabsadd L
.09143 .02382 -2.37764 .77254 Mabsadd L
.09143 .02382 -0.91448 -0.29713 Mabsadd L
.09143 .02382 -1.46946 -2.02254 Mabsadd L
.09143 .02382 0 -0.96154 Mabsadd L
.09143 .02382 1.46946 -2.02254 Mabsadd L
.09143 .02382 .91448 -0.29713 Mabsadd L
.09143 .02382 2.37764 .77254 Mabsadd L
.09143 .02382 .56518 .7779 Mabsadd L
.09143 .02382 0 2.5 Mabsadd L closepath F
.09238 .02382 0 2.5 Mabsadd m
.09238 .02382 -0.56518 .7779 Mabsadd L
.09238 .02382 -2.37764 .77254 Mabsadd L
.09238 .02382 -0.91448 -0.29713 Mabsadd L
.09238 .02382 -1.46946 -2.02254 Mabsadd L
.09238 .02382 0 -0.96154 Mabsadd L
.09238 .02382 1.46946 -2.02254 Mabsadd L
.09238 .02382 .91448 -0.29713 Mabsadd L
.09238 .02382 2.37764 .77254 Mabsadd L
.09238 .02382 .56518 .7779 Mabsadd L
.09238 .02382 0 2.5 Mabsadd L closepath F
.09333 .02381 0 2.5 Mabsadd m
.09333 .02381 -0.56518 .7779 Mabsadd L
.09333 .02381 -2.37764 .77254 Mabsadd L
.09333 .02381 -0.91448 -0.29713 Mabsadd L
.09333 .02381 -1.46946 -2.02254 Mabsadd L
.09333 .02381 0 -0.96154 Mabsadd L
.09333 .02381 1.46946 -2.02254 Mabsadd L
.09333 .02381 .91448 -0.29713 Mabsadd L
.09333 .02381 2.37764 .77254 Mabsadd L
.09333 .02381 .56518 .7779 Mabsadd L
.09333 .02381 0 2.5 Mabsadd L closepath F
.09429 .02382 0 2.5 Mabsadd m
.09429 .02382 -0.56518 .7779 Mabsadd L
.09429 .02382 -2.37764 .77254 Mabsadd L
.09429 .02382 -0.91448 -0.29713 Mabsadd L
.09429 .02382 -1.46946 -2.02254 Mabsadd L
.09429 .02382 0 -0.96154 Mabsadd L
.09429 .02382 1.46946 -2.02254 Mabsadd L
.09429 .02382 .91448 -0.29713 Mabsadd L
.09429 .02382 2.37764 .77254 Mabsadd L
.09429 .02382 .56518 .7779 Mabsadd L
.09429 .02382 0 2.5 Mabsadd L closepath F
.09524 .02381 0 2.5 Mabsadd m
.09524 .02381 -0.56518 .7779 Mabsadd L
.09524 .02381 -2.37764 .77254 Mabsadd L
.09524 .02381 -0.91448 -0.29713 Mabsadd L
.09524 .02381 -1.46946 -2.02254 Mabsadd L
.09524 .02381 0 -0.96154 Mabsadd L
.09524 .02381 1.46946 -2.02254 Mabsadd L
.09524 .02381 .91448 -0.29713 Mabsadd L
.09524 .02381 2.37764 .77254 Mabsadd L
.09524 .02381 .56518 .7779 Mabsadd L
.09524 .02381 0 2.5 Mabsadd L closepath F
.09619 .02381 0 2.5 Mabsadd m
.09619 .02381 -0.56518 .7779 Mabsadd L
.09619 .02381 -2.37764 .77254 Mabsadd L
.09619 .02381 -0.91448 -0.29713 Mabsadd L
.09619 .02381 -1.46946 -2.02254 Mabsadd L
.09619 .02381 0 -0.96154 Mabsadd L
.09619 .02381 1.46946 -2.02254 Mabsadd L
.09619 .02381 .91448 -0.29713 Mabsadd L
.09619 .02381 2.37764 .77254 Mabsadd L
.09619 .02381 .56518 .7779 Mabsadd L
.09619 .02381 0 2.5 Mabsadd L closepath F
.09714 .02382 0 2.5 Mabsadd m
.09714 .02382 -0.56518 .7779 Mabsadd L
.09714 .02382 -2.37764 .77254 Mabsadd L
.09714 .02382 -0.91448 -0.29713 Mabsadd L
.09714 .02382 -1.46946 -2.02254 Mabsadd L
.09714 .02382 0 -0.96154 Mabsadd L
.09714 .02382 1.46946 -2.02254 Mabsadd L
.09714 .02382 .91448 -0.29713 Mabsadd L
.09714 .02382 2.37764 .77254 Mabsadd L
.09714 .02382 .56518 .7779 Mabsadd L
.09714 .02382 0 2.5 Mabsadd L closepath F
.0981 .02382 0 2.5 Mabsadd m
.0981 .02382 -0.56518 .7779 Mabsadd L
.0981 .02382 -2.37764 .77254 Mabsadd L
.0981 .02382 -0.91448 -0.29713 Mabsadd L
.0981 .02382 -1.46946 -2.02254 Mabsadd L
.0981 .02382 0 -0.96154 Mabsadd L
.0981 .02382 1.46946 -2.02254 Mabsadd L
.0981 .02382 .91448 -0.29713 Mabsadd L
.0981 .02382 2.37764 .77254 Mabsadd L
.0981 .02382 .56518 .7779 Mabsadd L
.0981 .02382 0 2.5 Mabsadd L closepath F
.09905 .02382 0 2.5 Mabsadd m
.09905 .02382 -0.56518 .7779 Mabsadd L
.09905 .02382 -2.37764 .77254 Mabsadd L
.09905 .02382 -0.91448 -0.29713 Mabsadd L
.09905 .02382 -1.46946 -2.02254 Mabsadd L
.09905 .02382 0 -0.96154 Mabsadd L
.09905 .02382 1.46946 -2.02254 Mabsadd L
.09905 .02382 .91448 -0.29713 Mabsadd L
.09905 .02382 2.37764 .77254 Mabsadd L
.09905 .02382 .56518 .7779 Mabsadd L
.09905 .02382 0 2.5 Mabsadd L closepath F
.1 .02382 0 2.5 Mabsadd m
.1 .02382 -0.56518 .7779 Mabsadd L
.1 .02382 -2.37764 .77254 Mabsadd L
.1 .02382 -0.91448 -0.29713 Mabsadd L
.1 .02382 -1.46946 -2.02254 Mabsadd L
.1 .02382 0 -0.96154 Mabsadd L
.1 .02382 1.46946 -2.02254 Mabsadd L
.1 .02382 .91448 -0.29713 Mabsadd L
.1 .02382 2.37764 .77254 Mabsadd L
.1 .02382 .56518 .7779 Mabsadd L
.1 .02382 0 2.5 Mabsadd L closepath F
.10095 .02382 0 2.5 Mabsadd m
.10095 .02382 -0.56518 .7779 Mabsadd L
.10095 .02382 -2.37764 .77254 Mabsadd L
.10095 .02382 -0.91448 -0.29713 Mabsadd L
.10095 .02382 -1.46946 -2.02254 Mabsadd L
.10095 .02382 0 -0.96154 Mabsadd L
.10095 .02382 1.46946 -2.02254 Mabsadd L
.10095 .02382 .91448 -0.29713 Mabsadd L
.10095 .02382 2.37764 .77254 Mabsadd L
.10095 .02382 .56518 .7779 Mabsadd L
.10095 .02382 0 2.5 Mabsadd L closepath F
.1019 .02382 0 2.5 Mabsadd m
.1019 .02382 -0.56518 .7779 Mabsadd L
.1019 .02382 -2.37764 .77254 Mabsadd L
.1019 .02382 -0.91448 -0.29713 Mabsadd L
.1019 .02382 -1.46946 -2.02254 Mabsadd L
.1019 .02382 0 -0.96154 Mabsadd L
.1019 .02382 1.46946 -2.02254 Mabsadd L
.1019 .02382 .91448 -0.29713 Mabsadd L
.1019 .02382 2.37764 .77254 Mabsadd L
.1019 .02382 .56518 .7779 Mabsadd L
.1019 .02382 0 2.5 Mabsadd L closepath F
.10286 .02382 0 2.5 Mabsadd m
.10286 .02382 -0.56518 .7779 Mabsadd L
.10286 .02382 -2.37764 .77254 Mabsadd L
.10286 .02382 -0.91448 -0.29713 Mabsadd L
.10286 .02382 -1.46946 -2.02254 Mabsadd L
.10286 .02382 0 -0.96154 Mabsadd L
.10286 .02382 1.46946 -2.02254 Mabsadd L
.10286 .02382 .91448 -0.29713 Mabsadd L
.10286 .02382 2.37764 .77254 Mabsadd L
.10286 .02382 .56518 .7779 Mabsadd L
.10286 .02382 0 2.5 Mabsadd L closepath F
.10381 .02382 0 2.5 Mabsadd m
.10381 .02382 -0.56518 .7779 Mabsadd L
.10381 .02382 -2.37764 .77254 Mabsadd L
.10381 .02382 -0.91448 -0.29713 Mabsadd L
.10381 .02382 -1.46946 -2.02254 Mabsadd L
.10381 .02382 0 -0.96154 Mabsadd L
.10381 .02382 1.46946 -2.02254 Mabsadd L
.10381 .02382 .91448 -0.29713 Mabsadd L
.10381 .02382 2.37764 .77254 Mabsadd L
.10381 .02382 .56518 .7779 Mabsadd L
.10381 .02382 0 2.5 Mabsadd L closepath F
.10476 .02383 0 2.5 Mabsadd m
.10476 .02383 -0.56518 .7779 Mabsadd L
.10476 .02383 -2.37764 .77254 Mabsadd L
.10476 .02383 -0.91448 -0.29713 Mabsadd L
.10476 .02383 -1.46946 -2.02254 Mabsadd L
.10476 .02383 0 -0.96154 Mabsadd L
.10476 .02383 1.46946 -2.02254 Mabsadd L
.10476 .02383 .91448 -0.29713 Mabsadd L
.10476 .02383 2.37764 .77254 Mabsadd L
.10476 .02383 .56518 .7779 Mabsadd L
.10476 .02383 0 2.5 Mabsadd L closepath F
.10571 .02382 0 2.5 Mabsadd m
.10571 .02382 -0.56518 .7779 Mabsadd L
.10571 .02382 -2.37764 .77254 Mabsadd L
.10571 .02382 -0.91448 -0.29713 Mabsadd L
.10571 .02382 -1.46946 -2.02254 Mabsadd L
.10571 .02382 0 -0.96154 Mabsadd L
.10571 .02382 1.46946 -2.02254 Mabsadd L
.10571 .02382 .91448 -0.29713 Mabsadd L
.10571 .02382 2.37764 .77254 Mabsadd L
.10571 .02382 .56518 .7779 Mabsadd L
.10571 .02382 0 2.5 Mabsadd L closepath F
.10667 .02382 0 2.5 Mabsadd m
.10667 .02382 -0.56518 .7779 Mabsadd L
.10667 .02382 -2.37764 .77254 Mabsadd L
.10667 .02382 -0.91448 -0.29713 Mabsadd L
.10667 .02382 -1.46946 -2.02254 Mabsadd L
.10667 .02382 0 -0.96154 Mabsadd L
.10667 .02382 1.46946 -2.02254 Mabsadd L
.10667 .02382 .91448 -0.29713 Mabsadd L
.10667 .02382 2.37764 .77254 Mabsadd L
.10667 .02382 .56518 .7779 Mabsadd L
.10667 .02382 0 2.5 Mabsadd L closepath F
.10762 .02382 0 2.5 Mabsadd m
.10762 .02382 -0.56518 .7779 Mabsadd L
.10762 .02382 -2.37764 .77254 Mabsadd L
.10762 .02382 -0.91448 -0.29713 Mabsadd L
.10762 .02382 -1.46946 -2.02254 Mabsadd L
.10762 .02382 0 -0.96154 Mabsadd L
.10762 .02382 1.46946 -2.02254 Mabsadd L
.10762 .02382 .91448 -0.29713 Mabsadd L
.10762 .02382 2.37764 .77254 Mabsadd L
.10762 .02382 .56518 .7779 Mabsadd L
.10762 .02382 0 2.5 Mabsadd L closepath F
.10857 .02382 0 2.5 Mabsadd m
.10857 .02382 -0.56518 .7779 Mabsadd L
.10857 .02382 -2.37764 .77254 Mabsadd L
.10857 .02382 -0.91448 -0.29713 Mabsadd L
.10857 .02382 -1.46946 -2.02254 Mabsadd L
.10857 .02382 0 -0.96154 Mabsadd L
.10857 .02382 1.46946 -2.02254 Mabsadd L
.10857 .02382 .91448 -0.29713 Mabsadd L
.10857 .02382 2.37764 .77254 Mabsadd L
.10857 .02382 .56518 .7779 Mabsadd L
.10857 .02382 0 2.5 Mabsadd L closepath F
.10952 .02382 0 2.5 Mabsadd m
.10952 .02382 -0.56518 .7779 Mabsadd L
.10952 .02382 -2.37764 .77254 Mabsadd L
.10952 .02382 -0.91448 -0.29713 Mabsadd L
.10952 .02382 -1.46946 -2.02254 Mabsadd L
.10952 .02382 0 -0.96154 Mabsadd L
.10952 .02382 1.46946 -2.02254 Mabsadd L
.10952 .02382 .91448 -0.29713 Mabsadd L
.10952 .02382 2.37764 .77254 Mabsadd L
.10952 .02382 .56518 .7779 Mabsadd L
.10952 .02382 0 2.5 Mabsadd L closepath F
.11048 .02382 0 2.5 Mabsadd m
.11048 .02382 -0.56518 .7779 Mabsadd L
.11048 .02382 -2.37764 .77254 Mabsadd L
.11048 .02382 -0.91448 -0.29713 Mabsadd L
.11048 .02382 -1.46946 -2.02254 Mabsadd L
.11048 .02382 0 -0.96154 Mabsadd L
.11048 .02382 1.46946 -2.02254 Mabsadd L
.11048 .02382 .91448 -0.29713 Mabsadd L
.11048 .02382 2.37764 .77254 Mabsadd L
.11048 .02382 .56518 .7779 Mabsadd L
.11048 .02382 0 2.5 Mabsadd L closepath F
.11143 .02382 0 2.5 Mabsadd m
.11143 .02382 -0.56518 .7779 Mabsadd L
.11143 .02382 -2.37764 .77254 Mabsadd L
.11143 .02382 -0.91448 -0.29713 Mabsadd L
.11143 .02382 -1.46946 -2.02254 Mabsadd L
.11143 .02382 0 -0.96154 Mabsadd L
.11143 .02382 1.46946 -2.02254 Mabsadd L
.11143 .02382 .91448 -0.29713 Mabsadd L
.11143 .02382 2.37764 .77254 Mabsadd L
.11143 .02382 .56518 .7779 Mabsadd L
.11143 .02382 0 2.5 Mabsadd L closepath F
.11238 .02382 0 2.5 Mabsadd m
.11238 .02382 -0.56518 .7779 Mabsadd L
.11238 .02382 -2.37764 .77254 Mabsadd L
.11238 .02382 -0.91448 -0.29713 Mabsadd L
.11238 .02382 -1.46946 -2.02254 Mabsadd L
.11238 .02382 0 -0.96154 Mabsadd L
.11238 .02382 1.46946 -2.02254 Mabsadd L
.11238 .02382 .91448 -0.29713 Mabsadd L
.11238 .02382 2.37764 .77254 Mabsadd L
.11238 .02382 .56518 .7779 Mabsadd L
.11238 .02382 0 2.5 Mabsadd L closepath F
.11333 .02382 0 2.5 Mabsadd m
.11333 .02382 -0.56518 .7779 Mabsadd L
.11333 .02382 -2.37764 .77254 Mabsadd L
.11333 .02382 -0.91448 -0.29713 Mabsadd L
.11333 .02382 -1.46946 -2.02254 Mabsadd L
.11333 .02382 0 -0.96154 Mabsadd L
.11333 .02382 1.46946 -2.02254 Mabsadd L
.11333 .02382 .91448 -0.29713 Mabsadd L
.11333 .02382 2.37764 .77254 Mabsadd L
.11333 .02382 .56518 .7779 Mabsadd L
.11333 .02382 0 2.5 Mabsadd L closepath F
.11429 .02382 0 2.5 Mabsadd m
.11429 .02382 -0.56518 .7779 Mabsadd L
.11429 .02382 -2.37764 .77254 Mabsadd L
.11429 .02382 -0.91448 -0.29713 Mabsadd L
.11429 .02382 -1.46946 -2.02254 Mabsadd L
.11429 .02382 0 -0.96154 Mabsadd L
.11429 .02382 1.46946 -2.02254 Mabsadd L
.11429 .02382 .91448 -0.29713 Mabsadd L
.11429 .02382 2.37764 .77254 Mabsadd L
.11429 .02382 .56518 .7779 Mabsadd L
.11429 .02382 0 2.5 Mabsadd L closepath F
.11524 .02382 0 2.5 Mabsadd m
.11524 .02382 -0.56518 .7779 Mabsadd L
.11524 .02382 -2.37764 .77254 Mabsadd L
.11524 .02382 -0.91448 -0.29713 Mabsadd L
.11524 .02382 -1.46946 -2.02254 Mabsadd L
.11524 .02382 0 -0.96154 Mabsadd L
.11524 .02382 1.46946 -2.02254 Mabsadd L
.11524 .02382 .91448 -0.29713 Mabsadd L
.11524 .02382 2.37764 .77254 Mabsadd L
.11524 .02382 .56518 .7779 Mabsadd L
.11524 .02382 0 2.5 Mabsadd L closepath F
.11619 .02382 0 2.5 Mabsadd m
.11619 .02382 -0.56518 .7779 Mabsadd L
.11619 .02382 -2.37764 .77254 Mabsadd L
.11619 .02382 -0.91448 -0.29713 Mabsadd L
.11619 .02382 -1.46946 -2.02254 Mabsadd L
.11619 .02382 0 -0.96154 Mabsadd L
.11619 .02382 1.46946 -2.02254 Mabsadd L
.11619 .02382 .91448 -0.29713 Mabsadd L
.11619 .02382 2.37764 .77254 Mabsadd L
.11619 .02382 .56518 .7779 Mabsadd L
.11619 .02382 0 2.5 Mabsadd L closepath F
.11714 .02382 0 2.5 Mabsadd m
.11714 .02382 -0.56518 .7779 Mabsadd L
.11714 .02382 -2.37764 .77254 Mabsadd L
.11714 .02382 -0.91448 -0.29713 Mabsadd L
.11714 .02382 -1.46946 -2.02254 Mabsadd L
.11714 .02382 0 -0.96154 Mabsadd L
.11714 .02382 1.46946 -2.02254 Mabsadd L
.11714 .02382 .91448 -0.29713 Mabsadd L
.11714 .02382 2.37764 .77254 Mabsadd L
.11714 .02382 .56518 .7779 Mabsadd L
.11714 .02382 0 2.5 Mabsadd L closepath F
.1181 .02382 0 2.5 Mabsadd m
.1181 .02382 -0.56518 .7779 Mabsadd L
.1181 .02382 -2.37764 .77254 Mabsadd L
.1181 .02382 -0.91448 -0.29713 Mabsadd L
.1181 .02382 -1.46946 -2.02254 Mabsadd L
.1181 .02382 0 -0.96154 Mabsadd L
.1181 .02382 1.46946 -2.02254 Mabsadd L
.1181 .02382 .91448 -0.29713 Mabsadd L
.1181 .02382 2.37764 .77254 Mabsadd L
.1181 .02382 .56518 .7779 Mabsadd L
.1181 .02382 0 2.5 Mabsadd L closepath F
.11905 .02382 0 2.5 Mabsadd m
.11905 .02382 -0.56518 .7779 Mabsadd L
.11905 .02382 -2.37764 .77254 Mabsadd L
.11905 .02382 -0.91448 -0.29713 Mabsadd L
.11905 .02382 -1.46946 -2.02254 Mabsadd L
.11905 .02382 0 -0.96154 Mabsadd L
.11905 .02382 1.46946 -2.02254 Mabsadd L
.11905 .02382 .91448 -0.29713 Mabsadd L
.11905 .02382 2.37764 .77254 Mabsadd L
.11905 .02382 .56518 .7779 Mabsadd L
.11905 .02382 0 2.5 Mabsadd L closepath F
.12 .02382 0 2.5 Mabsadd m
.12 .02382 -0.56518 .7779 Mabsadd L
.12 .02382 -2.37764 .77254 Mabsadd L
.12 .02382 -0.91448 -0.29713 Mabsadd L
.12 .02382 -1.46946 -2.02254 Mabsadd L
.12 .02382 0 -0.96154 Mabsadd L
.12 .02382 1.46946 -2.02254 Mabsadd L
.12 .02382 .91448 -0.29713 Mabsadd L
.12 .02382 2.37764 .77254 Mabsadd L
.12 .02382 .56518 .7779 Mabsadd L
.12 .02382 0 2.5 Mabsadd L closepath F
.12095 .02382 0 2.5 Mabsadd m
.12095 .02382 -0.56518 .7779 Mabsadd L
.12095 .02382 -2.37764 .77254 Mabsadd L
.12095 .02382 -0.91448 -0.29713 Mabsadd L
.12095 .02382 -1.46946 -2.02254 Mabsadd L
.12095 .02382 0 -0.96154 Mabsadd L
.12095 .02382 1.46946 -2.02254 Mabsadd L
.12095 .02382 .91448 -0.29713 Mabsadd L
.12095 .02382 2.37764 .77254 Mabsadd L
.12095 .02382 .56518 .7779 Mabsadd L
.12095 .02382 0 2.5 Mabsadd L closepath F
.1219 .02382 0 2.5 Mabsadd m
.1219 .02382 -0.56518 .7779 Mabsadd L
.1219 .02382 -2.37764 .77254 Mabsadd L
.1219 .02382 -0.91448 -0.29713 Mabsadd L
.1219 .02382 -1.46946 -2.02254 Mabsadd L
.1219 .02382 0 -0.96154 Mabsadd L
.1219 .02382 1.46946 -2.02254 Mabsadd L
.1219 .02382 .91448 -0.29713 Mabsadd L
.1219 .02382 2.37764 .77254 Mabsadd L
.1219 .02382 .56518 .7779 Mabsadd L
.1219 .02382 0 2.5 Mabsadd L closepath F
.12286 .02382 0 2.5 Mabsadd m
.12286 .02382 -0.56518 .7779 Mabsadd L
.12286 .02382 -2.37764 .77254 Mabsadd L
.12286 .02382 -0.91448 -0.29713 Mabsadd L
.12286 .02382 -1.46946 -2.02254 Mabsadd L
.12286 .02382 0 -0.96154 Mabsadd L
.12286 .02382 1.46946 -2.02254 Mabsadd L
.12286 .02382 .91448 -0.29713 Mabsadd L
.12286 .02382 2.37764 .77254 Mabsadd L
.12286 .02382 .56518 .7779 Mabsadd L
.12286 .02382 0 2.5 Mabsadd L closepath F
.12381 .02382 0 2.5 Mabsadd m
.12381 .02382 -0.56518 .7779 Mabsadd L
.12381 .02382 -2.37764 .77254 Mabsadd L
.12381 .02382 -0.91448 -0.29713 Mabsadd L
.12381 .02382 -1.46946 -2.02254 Mabsadd L
.12381 .02382 0 -0.96154 Mabsadd L
.12381 .02382 1.46946 -2.02254 Mabsadd L
.12381 .02382 .91448 -0.29713 Mabsadd L
.12381 .02382 2.37764 .77254 Mabsadd L
.12381 .02382 .56518 .7779 Mabsadd L
.12381 .02382 0 2.5 Mabsadd L closepath F
.12476 .02382 0 2.5 Mabsadd m
.12476 .02382 -0.56518 .7779 Mabsadd L
.12476 .02382 -2.37764 .77254 Mabsadd L
.12476 .02382 -0.91448 -0.29713 Mabsadd L
.12476 .02382 -1.46946 -2.02254 Mabsadd L
.12476 .02382 0 -0.96154 Mabsadd L
.12476 .02382 1.46946 -2.02254 Mabsadd L
.12476 .02382 .91448 -0.29713 Mabsadd L
.12476 .02382 2.37764 .77254 Mabsadd L
.12476 .02382 .56518 .7779 Mabsadd L
.12476 .02382 0 2.5 Mabsadd L closepath F
.12571 .02382 0 2.5 Mabsadd m
.12571 .02382 -0.56518 .7779 Mabsadd L
.12571 .02382 -2.37764 .77254 Mabsadd L
.12571 .02382 -0.91448 -0.29713 Mabsadd L
.12571 .02382 -1.46946 -2.02254 Mabsadd L
.12571 .02382 0 -0.96154 Mabsadd L
.12571 .02382 1.46946 -2.02254 Mabsadd L
.12571 .02382 .91448 -0.29713 Mabsadd L
.12571 .02382 2.37764 .77254 Mabsadd L
.12571 .02382 .56518 .7779 Mabsadd L
.12571 .02382 0 2.5 Mabsadd L closepath F
.12667 .02382 0 2.5 Mabsadd m
.12667 .02382 -0.56518 .7779 Mabsadd L
.12667 .02382 -2.37764 .77254 Mabsadd L
.12667 .02382 -0.91448 -0.29713 Mabsadd L
.12667 .02382 -1.46946 -2.02254 Mabsadd L
.12667 .02382 0 -0.96154 Mabsadd L
.12667 .02382 1.46946 -2.02254 Mabsadd L
.12667 .02382 .91448 -0.29713 Mabsadd L
.12667 .02382 2.37764 .77254 Mabsadd L
.12667 .02382 .56518 .7779 Mabsadd L
.12667 .02382 0 2.5 Mabsadd L closepath F
.12762 .02382 0 2.5 Mabsadd m
.12762 .02382 -0.56518 .7779 Mabsadd L
.12762 .02382 -2.37764 .77254 Mabsadd L
.12762 .02382 -0.91448 -0.29713 Mabsadd L
.12762 .02382 -1.46946 -2.02254 Mabsadd L
.12762 .02382 0 -0.96154 Mabsadd L
.12762 .02382 1.46946 -2.02254 Mabsadd L
.12762 .02382 .91448 -0.29713 Mabsadd L
.12762 .02382 2.37764 .77254 Mabsadd L
.12762 .02382 .56518 .7779 Mabsadd L
.12762 .02382 0 2.5 Mabsadd L closepath F
.12857 .02382 0 2.5 Mabsadd m
.12857 .02382 -0.56518 .7779 Mabsadd L
.12857 .02382 -2.37764 .77254 Mabsadd L
.12857 .02382 -0.91448 -0.29713 Mabsadd L
.12857 .02382 -1.46946 -2.02254 Mabsadd L
.12857 .02382 0 -0.96154 Mabsadd L
.12857 .02382 1.46946 -2.02254 Mabsadd L
.12857 .02382 .91448 -0.29713 Mabsadd L
.12857 .02382 2.37764 .77254 Mabsadd L
.12857 .02382 .56518 .7779 Mabsadd L
.12857 .02382 0 2.5 Mabsadd L closepath F
.12952 .02382 0 2.5 Mabsadd m
.12952 .02382 -0.56518 .7779 Mabsadd L
.12952 .02382 -2.37764 .77254 Mabsadd L
.12952 .02382 -0.91448 -0.29713 Mabsadd L
.12952 .02382 -1.46946 -2.02254 Mabsadd L
.12952 .02382 0 -0.96154 Mabsadd L
.12952 .02382 1.46946 -2.02254 Mabsadd L
.12952 .02382 .91448 -0.29713 Mabsadd L
.12952 .02382 2.37764 .77254 Mabsadd L
.12952 .02382 .56518 .7779 Mabsadd L
.12952 .02382 0 2.5 Mabsadd L closepath F
.13048 .02382 0 2.5 Mabsadd m
.13048 .02382 -0.56518 .7779 Mabsadd L
.13048 .02382 -2.37764 .77254 Mabsadd L
.13048 .02382 -0.91448 -0.29713 Mabsadd L
.13048 .02382 -1.46946 -2.02254 Mabsadd L
.13048 .02382 0 -0.96154 Mabsadd L
.13048 .02382 1.46946 -2.02254 Mabsadd L
.13048 .02382 .91448 -0.29713 Mabsadd L
.13048 .02382 2.37764 .77254 Mabsadd L
.13048 .02382 .56518 .7779 Mabsadd L
.13048 .02382 0 2.5 Mabsadd L closepath F
.13143 .02382 0 2.5 Mabsadd m
.13143 .02382 -0.56518 .7779 Mabsadd L
.13143 .02382 -2.37764 .77254 Mabsadd L
.13143 .02382 -0.91448 -0.29713 Mabsadd L
.13143 .02382 -1.46946 -2.02254 Mabsadd L
.13143 .02382 0 -0.96154 Mabsadd L
.13143 .02382 1.46946 -2.02254 Mabsadd L
.13143 .02382 .91448 -0.29713 Mabsadd L
.13143 .02382 2.37764 .77254 Mabsadd L
.13143 .02382 .56518 .7779 Mabsadd L
.13143 .02382 0 2.5 Mabsadd L closepath F
.13238 .02382 0 2.5 Mabsadd m
.13238 .02382 -0.56518 .7779 Mabsadd L
.13238 .02382 -2.37764 .77254 Mabsadd L
.13238 .02382 -0.91448 -0.29713 Mabsadd L
.13238 .02382 -1.46946 -2.02254 Mabsadd L
.13238 .02382 0 -0.96154 Mabsadd L
.13238 .02382 1.46946 -2.02254 Mabsadd L
.13238 .02382 .91448 -0.29713 Mabsadd L
.13238 .02382 2.37764 .77254 Mabsadd L
.13238 .02382 .56518 .7779 Mabsadd L
.13238 .02382 0 2.5 Mabsadd L closepath F
.13333 .02382 0 2.5 Mabsadd m
.13333 .02382 -0.56518 .7779 Mabsadd L
.13333 .02382 -2.37764 .77254 Mabsadd L
.13333 .02382 -0.91448 -0.29713 Mabsadd L
.13333 .02382 -1.46946 -2.02254 Mabsadd L
.13333 .02382 0 -0.96154 Mabsadd L
.13333 .02382 1.46946 -2.02254 Mabsadd L
.13333 .02382 .91448 -0.29713 Mabsadd L
.13333 .02382 2.37764 .77254 Mabsadd L
.13333 .02382 .56518 .7779 Mabsadd L
.13333 .02382 0 2.5 Mabsadd L closepath F
.13429 .02382 0 2.5 Mabsadd m
.13429 .02382 -0.56518 .7779 Mabsadd L
.13429 .02382 -2.37764 .77254 Mabsadd L
.13429 .02382 -0.91448 -0.29713 Mabsadd L
.13429 .02382 -1.46946 -2.02254 Mabsadd L
.13429 .02382 0 -0.96154 Mabsadd L
.13429 .02382 1.46946 -2.02254 Mabsadd L
.13429 .02382 .91448 -0.29713 Mabsadd L
.13429 .02382 2.37764 .77254 Mabsadd L
.13429 .02382 .56518 .7779 Mabsadd L
.13429 .02382 0 2.5 Mabsadd L closepath F
.13524 .02382 0 2.5 Mabsadd m
.13524 .02382 -0.56518 .7779 Mabsadd L
.13524 .02382 -2.37764 .77254 Mabsadd L
.13524 .02382 -0.91448 -0.29713 Mabsadd L
.13524 .02382 -1.46946 -2.02254 Mabsadd L
.13524 .02382 0 -0.96154 Mabsadd L
.13524 .02382 1.46946 -2.02254 Mabsadd L
.13524 .02382 .91448 -0.29713 Mabsadd L
.13524 .02382 2.37764 .77254 Mabsadd L
.13524 .02382 .56518 .7779 Mabsadd L
.13524 .02382 0 2.5 Mabsadd L closepath F
.13619 .02382 0 2.5 Mabsadd m
.13619 .02382 -0.56518 .7779 Mabsadd L
.13619 .02382 -2.37764 .77254 Mabsadd L
.13619 .02382 -0.91448 -0.29713 Mabsadd L
.13619 .02382 -1.46946 -2.02254 Mabsadd L
.13619 .02382 0 -0.96154 Mabsadd L
.13619 .02382 1.46946 -2.02254 Mabsadd L
.13619 .02382 .91448 -0.29713 Mabsadd L
.13619 .02382 2.37764 .77254 Mabsadd L
.13619 .02382 .56518 .7779 Mabsadd L
.13619 .02382 0 2.5 Mabsadd L closepath F
.13714 .02382 0 2.5 Mabsadd m
.13714 .02382 -0.56518 .7779 Mabsadd L
.13714 .02382 -2.37764 .77254 Mabsadd L
.13714 .02382 -0.91448 -0.29713 Mabsadd L
.13714 .02382 -1.46946 -2.02254 Mabsadd L
.13714 .02382 0 -0.96154 Mabsadd L
.13714 .02382 1.46946 -2.02254 Mabsadd L
.13714 .02382 .91448 -0.29713 Mabsadd L
.13714 .02382 2.37764 .77254 Mabsadd L
.13714 .02382 .56518 .7779 Mabsadd L
.13714 .02382 0 2.5 Mabsadd L closepath F
.1381 .02382 0 2.5 Mabsadd m
.1381 .02382 -0.56518 .7779 Mabsadd L
.1381 .02382 -2.37764 .77254 Mabsadd L
.1381 .02382 -0.91448 -0.29713 Mabsadd L
.1381 .02382 -1.46946 -2.02254 Mabsadd L
.1381 .02382 0 -0.96154 Mabsadd L
.1381 .02382 1.46946 -2.02254 Mabsadd L
.1381 .02382 .91448 -0.29713 Mabsadd L
.1381 .02382 2.37764 .77254 Mabsadd L
.1381 .02382 .56518 .7779 Mabsadd L
.1381 .02382 0 2.5 Mabsadd L closepath F
.13905 .02382 0 2.5 Mabsadd m
.13905 .02382 -0.56518 .7779 Mabsadd L
.13905 .02382 -2.37764 .77254 Mabsadd L
.13905 .02382 -0.91448 -0.29713 Mabsadd L
.13905 .02382 -1.46946 -2.02254 Mabsadd L
.13905 .02382 0 -0.96154 Mabsadd L
.13905 .02382 1.46946 -2.02254 Mabsadd L
.13905 .02382 .91448 -0.29713 Mabsadd L
.13905 .02382 2.37764 .77254 Mabsadd L
.13905 .02382 .56518 .7779 Mabsadd L
.13905 .02382 0 2.5 Mabsadd L closepath F
.14 .02382 0 2.5 Mabsadd m
.14 .02382 -0.56518 .7779 Mabsadd L
.14 .02382 -2.37764 .77254 Mabsadd L
.14 .02382 -0.91448 -0.29713 Mabsadd L
.14 .02382 -1.46946 -2.02254 Mabsadd L
.14 .02382 0 -0.96154 Mabsadd L
.14 .02382 1.46946 -2.02254 Mabsadd L
.14 .02382 .91448 -0.29713 Mabsadd L
.14 .02382 2.37764 .77254 Mabsadd L
.14 .02382 .56518 .7779 Mabsadd L
.14 .02382 0 2.5 Mabsadd L closepath F
.14095 .02382 0 2.5 Mabsadd m
.14095 .02382 -0.56518 .7779 Mabsadd L
.14095 .02382 -2.37764 .77254 Mabsadd L
.14095 .02382 -0.91448 -0.29713 Mabsadd L
.14095 .02382 -1.46946 -2.02254 Mabsadd L
.14095 .02382 0 -0.96154 Mabsadd L
.14095 .02382 1.46946 -2.02254 Mabsadd L
.14095 .02382 .91448 -0.29713 Mabsadd L
.14095 .02382 2.37764 .77254 Mabsadd L
.14095 .02382 .56518 .7779 Mabsadd L
.14095 .02382 0 2.5 Mabsadd L closepath F
.1419 .02382 0 2.5 Mabsadd m
.1419 .02382 -0.56518 .7779 Mabsadd L
.1419 .02382 -2.37764 .77254 Mabsadd L
.1419 .02382 -0.91448 -0.29713 Mabsadd L
.1419 .02382 -1.46946 -2.02254 Mabsadd L
.1419 .02382 0 -0.96154 Mabsadd L
.1419 .02382 1.46946 -2.02254 Mabsadd L
.1419 .02382 .91448 -0.29713 Mabsadd L
.1419 .02382 2.37764 .77254 Mabsadd L
.1419 .02382 .56518 .7779 Mabsadd L
.1419 .02382 0 2.5 Mabsadd L closepath F
.14286 .02381 0 2.5 Mabsadd m
.14286 .02381 -0.56518 .7779 Mabsadd L
.14286 .02381 -2.37764 .77254 Mabsadd L
.14286 .02381 -0.91448 -0.29713 Mabsadd L
.14286 .02381 -1.46946 -2.02254 Mabsadd L
.14286 .02381 0 -0.96154 Mabsadd L
.14286 .02381 1.46946 -2.02254 Mabsadd L
.14286 .02381 .91448 -0.29713 Mabsadd L
.14286 .02381 2.37764 .77254 Mabsadd L
.14286 .02381 .56518 .7779 Mabsadd L
.14286 .02381 0 2.5 Mabsadd L closepath F
.14381 .02381 0 2.5 Mabsadd m
.14381 .02381 -0.56518 .7779 Mabsadd L
.14381 .02381 -2.37764 .77254 Mabsadd L
.14381 .02381 -0.91448 -0.29713 Mabsadd L
.14381 .02381 -1.46946 -2.02254 Mabsadd L
.14381 .02381 0 -0.96154 Mabsadd L
.14381 .02381 1.46946 -2.02254 Mabsadd L
.14381 .02381 .91448 -0.29713 Mabsadd L
.14381 .02381 2.37764 .77254 Mabsadd L
.14381 .02381 .56518 .7779 Mabsadd L
.14381 .02381 0 2.5 Mabsadd L closepath F
.14476 .02381 0 2.5 Mabsadd m
.14476 .02381 -0.56518 .7779 Mabsadd L
.14476 .02381 -2.37764 .77254 Mabsadd L
.14476 .02381 -0.91448 -0.29713 Mabsadd L
.14476 .02381 -1.46946 -2.02254 Mabsadd L
.14476 .02381 0 -0.96154 Mabsadd L
.14476 .02381 1.46946 -2.02254 Mabsadd L
.14476 .02381 .91448 -0.29713 Mabsadd L
.14476 .02381 2.37764 .77254 Mabsadd L
.14476 .02381 .56518 .7779 Mabsadd L
.14476 .02381 0 2.5 Mabsadd L closepath F
.14571 .02381 0 2.5 Mabsadd m
.14571 .02381 -0.56518 .7779 Mabsadd L
.14571 .02381 -2.37764 .77254 Mabsadd L
.14571 .02381 -0.91448 -0.29713 Mabsadd L
.14571 .02381 -1.46946 -2.02254 Mabsadd L
.14571 .02381 0 -0.96154 Mabsadd L
.14571 .02381 1.46946 -2.02254 Mabsadd L
.14571 .02381 .91448 -0.29713 Mabsadd L
.14571 .02381 2.37764 .77254 Mabsadd L
.14571 .02381 .56518 .7779 Mabsadd L
.14571 .02381 0 2.5 Mabsadd L closepath F
.14667 .02381 0 2.5 Mabsadd m
.14667 .02381 -0.56518 .7779 Mabsadd L
.14667 .02381 -2.37764 .77254 Mabsadd L
.14667 .02381 -0.91448 -0.29713 Mabsadd L
.14667 .02381 -1.46946 -2.02254 Mabsadd L
.14667 .02381 0 -0.96154 Mabsadd L
.14667 .02381 1.46946 -2.02254 Mabsadd L
.14667 .02381 .91448 -0.29713 Mabsadd L
.14667 .02381 2.37764 .77254 Mabsadd L
.14667 .02381 .56518 .7779 Mabsadd L
.14667 .02381 0 2.5 Mabsadd L closepath F
.14762 .02381 0 2.5 Mabsadd m
.14762 .02381 -0.56518 .7779 Mabsadd L
.14762 .02381 -2.37764 .77254 Mabsadd L
.14762 .02381 -0.91448 -0.29713 Mabsadd L
.14762 .02381 -1.46946 -2.02254 Mabsadd L
.14762 .02381 0 -0.96154 Mabsadd L
.14762 .02381 1.46946 -2.02254 Mabsadd L
.14762 .02381 .91448 -0.29713 Mabsadd L
.14762 .02381 2.37764 .77254 Mabsadd L
.14762 .02381 .56518 .7779 Mabsadd L
.14762 .02381 0 2.5 Mabsadd L closepath F
.14857 .02381 0 2.5 Mabsadd m
.14857 .02381 -0.56518 .7779 Mabsadd L
.14857 .02381 -2.37764 .77254 Mabsadd L
.14857 .02381 -0.91448 -0.29713 Mabsadd L
.14857 .02381 -1.46946 -2.02254 Mabsadd L
.14857 .02381 0 -0.96154 Mabsadd L
.14857 .02381 1.46946 -2.02254 Mabsadd L
.14857 .02381 .91448 -0.29713 Mabsadd L
.14857 .02381 2.37764 .77254 Mabsadd L
.14857 .02381 .56518 .7779 Mabsadd L
.14857 .02381 0 2.5 Mabsadd L closepath F
.14952 .02381 0 2.5 Mabsadd m
.14952 .02381 -0.56518 .7779 Mabsadd L
.14952 .02381 -2.37764 .77254 Mabsadd L
.14952 .02381 -0.91448 -0.29713 Mabsadd L
.14952 .02381 -1.46946 -2.02254 Mabsadd L
.14952 .02381 0 -0.96154 Mabsadd L
.14952 .02381 1.46946 -2.02254 Mabsadd L
.14952 .02381 .91448 -0.29713 Mabsadd L
.14952 .02381 2.37764 .77254 Mabsadd L
.14952 .02381 .56518 .7779 Mabsadd L
.14952 .02381 0 2.5 Mabsadd L closepath F
.15048 .02381 0 2.5 Mabsadd m
.15048 .02381 -0.56518 .7779 Mabsadd L
.15048 .02381 -2.37764 .77254 Mabsadd L
.15048 .02381 -0.91448 -0.29713 Mabsadd L
.15048 .02381 -1.46946 -2.02254 Mabsadd L
.15048 .02381 0 -0.96154 Mabsadd L
.15048 .02381 1.46946 -2.02254 Mabsadd L
.15048 .02381 .91448 -0.29713 Mabsadd L
.15048 .02381 2.37764 .77254 Mabsadd L
.15048 .02381 .56518 .7779 Mabsadd L
.15048 .02381 0 2.5 Mabsadd L closepath F
.15143 .02381 0 2.5 Mabsadd m
.15143 .02381 -0.56518 .7779 Mabsadd L
.15143 .02381 -2.37764 .77254 Mabsadd L
.15143 .02381 -0.91448 -0.29713 Mabsadd L
.15143 .02381 -1.46946 -2.02254 Mabsadd L
.15143 .02381 0 -0.96154 Mabsadd L
.15143 .02381 1.46946 -2.02254 Mabsadd L
.15143 .02381 .91448 -0.29713 Mabsadd L
.15143 .02381 2.37764 .77254 Mabsadd L
.15143 .02381 .56518 .7779 Mabsadd L
.15143 .02381 0 2.5 Mabsadd L closepath F
.15238 .02381 0 2.5 Mabsadd m
.15238 .02381 -0.56518 .7779 Mabsadd L
.15238 .02381 -2.37764 .77254 Mabsadd L
.15238 .02381 -0.91448 -0.29713 Mabsadd L
.15238 .02381 -1.46946 -2.02254 Mabsadd L
.15238 .02381 0 -0.96154 Mabsadd L
.15238 .02381 1.46946 -2.02254 Mabsadd L
.15238 .02381 .91448 -0.29713 Mabsadd L
.15238 .02381 2.37764 .77254 Mabsadd L
.15238 .02381 .56518 .7779 Mabsadd L
.15238 .02381 0 2.5 Mabsadd L closepath F
.15333 .02381 0 2.5 Mabsadd m
.15333 .02381 -0.56518 .7779 Mabsadd L
.15333 .02381 -2.37764 .77254 Mabsadd L
.15333 .02381 -0.91448 -0.29713 Mabsadd L
.15333 .02381 -1.46946 -2.02254 Mabsadd L
.15333 .02381 0 -0.96154 Mabsadd L
.15333 .02381 1.46946 -2.02254 Mabsadd L
.15333 .02381 .91448 -0.29713 Mabsadd L
.15333 .02381 2.37764 .77254 Mabsadd L
.15333 .02381 .56518 .7779 Mabsadd L
.15333 .02381 0 2.5 Mabsadd L closepath F
.15429 .02381 0 2.5 Mabsadd m
.15429 .02381 -0.56518 .7779 Mabsadd L
.15429 .02381 -2.37764 .77254 Mabsadd L
.15429 .02381 -0.91448 -0.29713 Mabsadd L
.15429 .02381 -1.46946 -2.02254 Mabsadd L
.15429 .02381 0 -0.96154 Mabsadd L
.15429 .02381 1.46946 -2.02254 Mabsadd L
.15429 .02381 .91448 -0.29713 Mabsadd L
.15429 .02381 2.37764 .77254 Mabsadd L
.15429 .02381 .56518 .7779 Mabsadd L
.15429 .02381 0 2.5 Mabsadd L closepath F
.15524 .02381 0 2.5 Mabsadd m
.15524 .02381 -0.56518 .7779 Mabsadd L
.15524 .02381 -2.37764 .77254 Mabsadd L
.15524 .02381 -0.91448 -0.29713 Mabsadd L
.15524 .02381 -1.46946 -2.02254 Mabsadd L
.15524 .02381 0 -0.96154 Mabsadd L
.15524 .02381 1.46946 -2.02254 Mabsadd L
.15524 .02381 .91448 -0.29713 Mabsadd L
.15524 .02381 2.37764 .77254 Mabsadd L
.15524 .02381 .56518 .7779 Mabsadd L
.15524 .02381 0 2.5 Mabsadd L closepath F
.15619 .02381 0 2.5 Mabsadd m
.15619 .02381 -0.56518 .7779 Mabsadd L
.15619 .02381 -2.37764 .77254 Mabsadd L
.15619 .02381 -0.91448 -0.29713 Mabsadd L
.15619 .02381 -1.46946 -2.02254 Mabsadd L
.15619 .02381 0 -0.96154 Mabsadd L
.15619 .02381 1.46946 -2.02254 Mabsadd L
.15619 .02381 .91448 -0.29713 Mabsadd L
.15619 .02381 2.37764 .77254 Mabsadd L
.15619 .02381 .56518 .7779 Mabsadd L
.15619 .02381 0 2.5 Mabsadd L closepath F
.15714 .02381 0 2.5 Mabsadd m
.15714 .02381 -0.56518 .7779 Mabsadd L
.15714 .02381 -2.37764 .77254 Mabsadd L
.15714 .02381 -0.91448 -0.29713 Mabsadd L
.15714 .02381 -1.46946 -2.02254 Mabsadd L
.15714 .02381 0 -0.96154 Mabsadd L
.15714 .02381 1.46946 -2.02254 Mabsadd L
.15714 .02381 .91448 -0.29713 Mabsadd L
.15714 .02381 2.37764 .77254 Mabsadd L
.15714 .02381 .56518 .7779 Mabsadd L
.15714 .02381 0 2.5 Mabsadd L closepath F
.1581 .02381 0 2.5 Mabsadd m
.1581 .02381 -0.56518 .7779 Mabsadd L
.1581 .02381 -2.37764 .77254 Mabsadd L
.1581 .02381 -0.91448 -0.29713 Mabsadd L
.1581 .02381 -1.46946 -2.02254 Mabsadd L
.1581 .02381 0 -0.96154 Mabsadd L
.1581 .02381 1.46946 -2.02254 Mabsadd L
.1581 .02381 .91448 -0.29713 Mabsadd L
.1581 .02381 2.37764 .77254 Mabsadd L
.1581 .02381 .56518 .7779 Mabsadd L
.1581 .02381 0 2.5 Mabsadd L closepath F
.15905 .02381 0 2.5 Mabsadd m
.15905 .02381 -0.56518 .7779 Mabsadd L
.15905 .02381 -2.37764 .77254 Mabsadd L
.15905 .02381 -0.91448 -0.29713 Mabsadd L
.15905 .02381 -1.46946 -2.02254 Mabsadd L
.15905 .02381 0 -0.96154 Mabsadd L
.15905 .02381 1.46946 -2.02254 Mabsadd L
.15905 .02381 .91448 -0.29713 Mabsadd L
.15905 .02381 2.37764 .77254 Mabsadd L
.15905 .02381 .56518 .7779 Mabsadd L
.15905 .02381 0 2.5 Mabsadd L closepath F
.16 .02381 0 2.5 Mabsadd m
.16 .02381 -0.56518 .7779 Mabsadd L
.16 .02381 -2.37764 .77254 Mabsadd L
.16 .02381 -0.91448 -0.29713 Mabsadd L
.16 .02381 -1.46946 -2.02254 Mabsadd L
.16 .02381 0 -0.96154 Mabsadd L
.16 .02381 1.46946 -2.02254 Mabsadd L
.16 .02381 .91448 -0.29713 Mabsadd L
.16 .02381 2.37764 .77254 Mabsadd L
.16 .02381 .56518 .7779 Mabsadd L
.16 .02381 0 2.5 Mabsadd L closepath F
.16095 .02381 0 2.5 Mabsadd m
.16095 .02381 -0.56518 .7779 Mabsadd L
.16095 .02381 -2.37764 .77254 Mabsadd L
.16095 .02381 -0.91448 -0.29713 Mabsadd L
.16095 .02381 -1.46946 -2.02254 Mabsadd L
.16095 .02381 0 -0.96154 Mabsadd L
.16095 .02381 1.46946 -2.02254 Mabsadd L
.16095 .02381 .91448 -0.29713 Mabsadd L
.16095 .02381 2.37764 .77254 Mabsadd L
.16095 .02381 .56518 .7779 Mabsadd L
.16095 .02381 0 2.5 Mabsadd L closepath F
.1619 .02381 0 2.5 Mabsadd m
.1619 .02381 -0.56518 .7779 Mabsadd L
.1619 .02381 -2.37764 .77254 Mabsadd L
.1619 .02381 -0.91448 -0.29713 Mabsadd L
.1619 .02381 -1.46946 -2.02254 Mabsadd L
.1619 .02381 0 -0.96154 Mabsadd L
.1619 .02381 1.46946 -2.02254 Mabsadd L
.1619 .02381 .91448 -0.29713 Mabsadd L
.1619 .02381 2.37764 .77254 Mabsadd L
.1619 .02381 .56518 .7779 Mabsadd L
.1619 .02381 0 2.5 Mabsadd L closepath F
.16286 .02381 0 2.5 Mabsadd m
.16286 .02381 -0.56518 .7779 Mabsadd L
.16286 .02381 -2.37764 .77254 Mabsadd L
.16286 .02381 -0.91448 -0.29713 Mabsadd L
.16286 .02381 -1.46946 -2.02254 Mabsadd L
.16286 .02381 0 -0.96154 Mabsadd L
.16286 .02381 1.46946 -2.02254 Mabsadd L
.16286 .02381 .91448 -0.29713 Mabsadd L
.16286 .02381 2.37764 .77254 Mabsadd L
.16286 .02381 .56518 .7779 Mabsadd L
.16286 .02381 0 2.5 Mabsadd L closepath F
.16381 .02381 0 2.5 Mabsadd m
.16381 .02381 -0.56518 .7779 Mabsadd L
.16381 .02381 -2.37764 .77254 Mabsadd L
.16381 .02381 -0.91448 -0.29713 Mabsadd L
.16381 .02381 -1.46946 -2.02254 Mabsadd L
.16381 .02381 0 -0.96154 Mabsadd L
.16381 .02381 1.46946 -2.02254 Mabsadd L
.16381 .02381 .91448 -0.29713 Mabsadd L
.16381 .02381 2.37764 .77254 Mabsadd L
.16381 .02381 .56518 .7779 Mabsadd L
.16381 .02381 0 2.5 Mabsadd L closepath F
.16476 .02381 0 2.5 Mabsadd m
.16476 .02381 -0.56518 .7779 Mabsadd L
.16476 .02381 -2.37764 .77254 Mabsadd L
.16476 .02381 -0.91448 -0.29713 Mabsadd L
.16476 .02381 -1.46946 -2.02254 Mabsadd L
.16476 .02381 0 -0.96154 Mabsadd L
.16476 .02381 1.46946 -2.02254 Mabsadd L
.16476 .02381 .91448 -0.29713 Mabsadd L
.16476 .02381 2.37764 .77254 Mabsadd L
.16476 .02381 .56518 .7779 Mabsadd L
.16476 .02381 0 2.5 Mabsadd L closepath F
.16571 .02381 0 2.5 Mabsadd m
.16571 .02381 -0.56518 .7779 Mabsadd L
.16571 .02381 -2.37764 .77254 Mabsadd L
.16571 .02381 -0.91448 -0.29713 Mabsadd L
.16571 .02381 -1.46946 -2.02254 Mabsadd L
.16571 .02381 0 -0.96154 Mabsadd L
.16571 .02381 1.46946 -2.02254 Mabsadd L
.16571 .02381 .91448 -0.29713 Mabsadd L
.16571 .02381 2.37764 .77254 Mabsadd L
.16571 .02381 .56518 .7779 Mabsadd L
.16571 .02381 0 2.5 Mabsadd L closepath F
.16667 .02381 0 2.5 Mabsadd m
.16667 .02381 -0.56518 .7779 Mabsadd L
.16667 .02381 -2.37764 .77254 Mabsadd L
.16667 .02381 -0.91448 -0.29713 Mabsadd L
.16667 .02381 -1.46946 -2.02254 Mabsadd L
.16667 .02381 0 -0.96154 Mabsadd L
.16667 .02381 1.46946 -2.02254 Mabsadd L
.16667 .02381 .91448 -0.29713 Mabsadd L
.16667 .02381 2.37764 .77254 Mabsadd L
.16667 .02381 .56518 .7779 Mabsadd L
.16667 .02381 0 2.5 Mabsadd L closepath F
.16762 .02381 0 2.5 Mabsadd m
.16762 .02381 -0.56518 .7779 Mabsadd L
.16762 .02381 -2.37764 .77254 Mabsadd L
.16762 .02381 -0.91448 -0.29713 Mabsadd L
.16762 .02381 -1.46946 -2.02254 Mabsadd L
.16762 .02381 0 -0.96154 Mabsadd L
.16762 .02381 1.46946 -2.02254 Mabsadd L
.16762 .02381 .91448 -0.29713 Mabsadd L
.16762 .02381 2.37764 .77254 Mabsadd L
.16762 .02381 .56518 .7779 Mabsadd L
.16762 .02381 0 2.5 Mabsadd L closepath F
.16857 .02381 0 2.5 Mabsadd m
.16857 .02381 -0.56518 .7779 Mabsadd L
.16857 .02381 -2.37764 .77254 Mabsadd L
.16857 .02381 -0.91448 -0.29713 Mabsadd L
.16857 .02381 -1.46946 -2.02254 Mabsadd L
.16857 .02381 0 -0.96154 Mabsadd L
.16857 .02381 1.46946 -2.02254 Mabsadd L
.16857 .02381 .91448 -0.29713 Mabsadd L
.16857 .02381 2.37764 .77254 Mabsadd L
.16857 .02381 .56518 .7779 Mabsadd L
.16857 .02381 0 2.5 Mabsadd L closepath F
.16952 .02381 0 2.5 Mabsadd m
.16952 .02381 -0.56518 .7779 Mabsadd L
.16952 .02381 -2.37764 .77254 Mabsadd L
.16952 .02381 -0.91448 -0.29713 Mabsadd L
.16952 .02381 -1.46946 -2.02254 Mabsadd L
.16952 .02381 0 -0.96154 Mabsadd L
.16952 .02381 1.46946 -2.02254 Mabsadd L
.16952 .02381 .91448 -0.29713 Mabsadd L
.16952 .02381 2.37764 .77254 Mabsadd L
.16952 .02381 .56518 .7779 Mabsadd L
.16952 .02381 0 2.5 Mabsadd L closepath F
.17048 .02381 0 2.5 Mabsadd m
.17048 .02381 -0.56518 .7779 Mabsadd L
.17048 .02381 -2.37764 .77254 Mabsadd L
.17048 .02381 -0.91448 -0.29713 Mabsadd L
.17048 .02381 -1.46946 -2.02254 Mabsadd L
.17048 .02381 0 -0.96154 Mabsadd L
.17048 .02381 1.46946 -2.02254 Mabsadd L
.17048 .02381 .91448 -0.29713 Mabsadd L
.17048 .02381 2.37764 .77254 Mabsadd L
.17048 .02381 .56518 .7779 Mabsadd L
.17048 .02381 0 2.5 Mabsadd L closepath F
.17143 .02381 0 2.5 Mabsadd m
.17143 .02381 -0.56518 .7779 Mabsadd L
.17143 .02381 -2.37764 .77254 Mabsadd L
.17143 .02381 -0.91448 -0.29713 Mabsadd L
.17143 .02381 -1.46946 -2.02254 Mabsadd L
.17143 .02381 0 -0.96154 Mabsadd L
.17143 .02381 1.46946 -2.02254 Mabsadd L
.17143 .02381 .91448 -0.29713 Mabsadd L
.17143 .02381 2.37764 .77254 Mabsadd L
.17143 .02381 .56518 .7779 Mabsadd L
.17143 .02381 0 2.5 Mabsadd L closepath F
.17238 .02381 0 2.5 Mabsadd m
.17238 .02381 -0.56518 .7779 Mabsadd L
.17238 .02381 -2.37764 .77254 Mabsadd L
.17238 .02381 -0.91448 -0.29713 Mabsadd L
.17238 .02381 -1.46946 -2.02254 Mabsadd L
.17238 .02381 0 -0.96154 Mabsadd L
.17238 .02381 1.46946 -2.02254 Mabsadd L
.17238 .02381 .91448 -0.29713 Mabsadd L
.17238 .02381 2.37764 .77254 Mabsadd L
.17238 .02381 .56518 .7779 Mabsadd L
.17238 .02381 0 2.5 Mabsadd L closepath F
.17333 .02381 0 2.5 Mabsadd m
.17333 .02381 -0.56518 .7779 Mabsadd L
.17333 .02381 -2.37764 .77254 Mabsadd L
.17333 .02381 -0.91448 -0.29713 Mabsadd L
.17333 .02381 -1.46946 -2.02254 Mabsadd L
.17333 .02381 0 -0.96154 Mabsadd L
.17333 .02381 1.46946 -2.02254 Mabsadd L
.17333 .02381 .91448 -0.29713 Mabsadd L
.17333 .02381 2.37764 .77254 Mabsadd L
.17333 .02381 .56518 .7779 Mabsadd L
.17333 .02381 0 2.5 Mabsadd L closepath F
.17429 .02381 0 2.5 Mabsadd m
.17429 .02381 -0.56518 .7779 Mabsadd L
.17429 .02381 -2.37764 .77254 Mabsadd L
.17429 .02381 -0.91448 -0.29713 Mabsadd L
.17429 .02381 -1.46946 -2.02254 Mabsadd L
.17429 .02381 0 -0.96154 Mabsadd L
.17429 .02381 1.46946 -2.02254 Mabsadd L
.17429 .02381 .91448 -0.29713 Mabsadd L
.17429 .02381 2.37764 .77254 Mabsadd L
.17429 .02381 .56518 .7779 Mabsadd L
.17429 .02381 0 2.5 Mabsadd L closepath F
.17524 .02381 0 2.5 Mabsadd m
.17524 .02381 -0.56518 .7779 Mabsadd L
.17524 .02381 -2.37764 .77254 Mabsadd L
.17524 .02381 -0.91448 -0.29713 Mabsadd L
.17524 .02381 -1.46946 -2.02254 Mabsadd L
.17524 .02381 0 -0.96154 Mabsadd L
.17524 .02381 1.46946 -2.02254 Mabsadd L
.17524 .02381 .91448 -0.29713 Mabsadd L
.17524 .02381 2.37764 .77254 Mabsadd L
.17524 .02381 .56518 .7779 Mabsadd L
.17524 .02381 0 2.5 Mabsadd L closepath F
.17619 .02381 0 2.5 Mabsadd m
.17619 .02381 -0.56518 .7779 Mabsadd L
.17619 .02381 -2.37764 .77254 Mabsadd L
.17619 .02381 -0.91448 -0.29713 Mabsadd L
.17619 .02381 -1.46946 -2.02254 Mabsadd L
.17619 .02381 0 -0.96154 Mabsadd L
.17619 .02381 1.46946 -2.02254 Mabsadd L
.17619 .02381 .91448 -0.29713 Mabsadd L
.17619 .02381 2.37764 .77254 Mabsadd L
.17619 .02381 .56518 .7779 Mabsadd L
.17619 .02381 0 2.5 Mabsadd L closepath F
.17714 .02381 0 2.5 Mabsadd m
.17714 .02381 -0.56518 .7779 Mabsadd L
.17714 .02381 -2.37764 .77254 Mabsadd L
.17714 .02381 -0.91448 -0.29713 Mabsadd L
.17714 .02381 -1.46946 -2.02254 Mabsadd L
.17714 .02381 0 -0.96154 Mabsadd L
.17714 .02381 1.46946 -2.02254 Mabsadd L
.17714 .02381 .91448 -0.29713 Mabsadd L
.17714 .02381 2.37764 .77254 Mabsadd L
.17714 .02381 .56518 .7779 Mabsadd L
.17714 .02381 0 2.5 Mabsadd L closepath F
.1781 .02381 0 2.5 Mabsadd m
.1781 .02381 -0.56518 .7779 Mabsadd L
.1781 .02381 -2.37764 .77254 Mabsadd L
.1781 .02381 -0.91448 -0.29713 Mabsadd L
.1781 .02381 -1.46946 -2.02254 Mabsadd L
.1781 .02381 0 -0.96154 Mabsadd L
.1781 .02381 1.46946 -2.02254 Mabsadd L
.1781 .02381 .91448 -0.29713 Mabsadd L
.1781 .02381 2.37764 .77254 Mabsadd L
.1781 .02381 .56518 .7779 Mabsadd L
.1781 .02381 0 2.5 Mabsadd L closepath F
.17905 .02381 0 2.5 Mabsadd m
.17905 .02381 -0.56518 .7779 Mabsadd L
.17905 .02381 -2.37764 .77254 Mabsadd L
.17905 .02381 -0.91448 -0.29713 Mabsadd L
.17905 .02381 -1.46946 -2.02254 Mabsadd L
.17905 .02381 0 -0.96154 Mabsadd L
.17905 .02381 1.46946 -2.02254 Mabsadd L
.17905 .02381 .91448 -0.29713 Mabsadd L
.17905 .02381 2.37764 .77254 Mabsadd L
.17905 .02381 .56518 .7779 Mabsadd L
.17905 .02381 0 2.5 Mabsadd L closepath F
.18 .02381 0 2.5 Mabsadd m
.18 .02381 -0.56518 .7779 Mabsadd L
.18 .02381 -2.37764 .77254 Mabsadd L
.18 .02381 -0.91448 -0.29713 Mabsadd L
.18 .02381 -1.46946 -2.02254 Mabsadd L
.18 .02381 0 -0.96154 Mabsadd L
.18 .02381 1.46946 -2.02254 Mabsadd L
.18 .02381 .91448 -0.29713 Mabsadd L
.18 .02381 2.37764 .77254 Mabsadd L
.18 .02381 .56518 .7779 Mabsadd L
.18 .02381 0 2.5 Mabsadd L closepath F
.18095 .02381 0 2.5 Mabsadd m
.18095 .02381 -0.56518 .7779 Mabsadd L
.18095 .02381 -2.37764 .77254 Mabsadd L
.18095 .02381 -0.91448 -0.29713 Mabsadd L
.18095 .02381 -1.46946 -2.02254 Mabsadd L
.18095 .02381 0 -0.96154 Mabsadd L
.18095 .02381 1.46946 -2.02254 Mabsadd L
.18095 .02381 .91448 -0.29713 Mabsadd L
.18095 .02381 2.37764 .77254 Mabsadd L
.18095 .02381 .56518 .7779 Mabsadd L
.18095 .02381 0 2.5 Mabsadd L closepath F
.1819 .02381 0 2.5 Mabsadd m
.1819 .02381 -0.56518 .7779 Mabsadd L
.1819 .02381 -2.37764 .77254 Mabsadd L
.1819 .02381 -0.91448 -0.29713 Mabsadd L
.1819 .02381 -1.46946 -2.02254 Mabsadd L
.1819 .02381 0 -0.96154 Mabsadd L
.1819 .02381 1.46946 -2.02254 Mabsadd L
.1819 .02381 .91448 -0.29713 Mabsadd L
.1819 .02381 2.37764 .77254 Mabsadd L
.1819 .02381 .56518 .7779 Mabsadd L
.1819 .02381 0 2.5 Mabsadd L closepath F
.18286 .02381 0 2.5 Mabsadd m
.18286 .02381 -0.56518 .7779 Mabsadd L
.18286 .02381 -2.37764 .77254 Mabsadd L
.18286 .02381 -0.91448 -0.29713 Mabsadd L
.18286 .02381 -1.46946 -2.02254 Mabsadd L
.18286 .02381 0 -0.96154 Mabsadd L
.18286 .02381 1.46946 -2.02254 Mabsadd L
.18286 .02381 .91448 -0.29713 Mabsadd L
.18286 .02381 2.37764 .77254 Mabsadd L
.18286 .02381 .56518 .7779 Mabsadd L
.18286 .02381 0 2.5 Mabsadd L closepath F
.18381 .02381 0 2.5 Mabsadd m
.18381 .02381 -0.56518 .7779 Mabsadd L
.18381 .02381 -2.37764 .77254 Mabsadd L
.18381 .02381 -0.91448 -0.29713 Mabsadd L
.18381 .02381 -1.46946 -2.02254 Mabsadd L
.18381 .02381 0 -0.96154 Mabsadd L
.18381 .02381 1.46946 -2.02254 Mabsadd L
.18381 .02381 .91448 -0.29713 Mabsadd L
.18381 .02381 2.37764 .77254 Mabsadd L
.18381 .02381 .56518 .7779 Mabsadd L
.18381 .02381 0 2.5 Mabsadd L closepath F
.18476 .02381 0 2.5 Mabsadd m
.18476 .02381 -0.56518 .7779 Mabsadd L
.18476 .02381 -2.37764 .77254 Mabsadd L
.18476 .02381 -0.91448 -0.29713 Mabsadd L
.18476 .02381 -1.46946 -2.02254 Mabsadd L
.18476 .02381 0 -0.96154 Mabsadd L
.18476 .02381 1.46946 -2.02254 Mabsadd L
.18476 .02381 .91448 -0.29713 Mabsadd L
.18476 .02381 2.37764 .77254 Mabsadd L
.18476 .02381 .56518 .7779 Mabsadd L
.18476 .02381 0 2.5 Mabsadd L closepath F
.18571 .02381 0 2.5 Mabsadd m
.18571 .02381 -0.56518 .7779 Mabsadd L
.18571 .02381 -2.37764 .77254 Mabsadd L
.18571 .02381 -0.91448 -0.29713 Mabsadd L
.18571 .02381 -1.46946 -2.02254 Mabsadd L
.18571 .02381 0 -0.96154 Mabsadd L
.18571 .02381 1.46946 -2.02254 Mabsadd L
.18571 .02381 .91448 -0.29713 Mabsadd L
.18571 .02381 2.37764 .77254 Mabsadd L
.18571 .02381 .56518 .7779 Mabsadd L
.18571 .02381 0 2.5 Mabsadd L closepath F
.18667 .02381 0 2.5 Mabsadd m
.18667 .02381 -0.56518 .7779 Mabsadd L
.18667 .02381 -2.37764 .77254 Mabsadd L
.18667 .02381 -0.91448 -0.29713 Mabsadd L
.18667 .02381 -1.46946 -2.02254 Mabsadd L
.18667 .02381 0 -0.96154 Mabsadd L
.18667 .02381 1.46946 -2.02254 Mabsadd L
.18667 .02381 .91448 -0.29713 Mabsadd L
.18667 .02381 2.37764 .77254 Mabsadd L
.18667 .02381 .56518 .7779 Mabsadd L
.18667 .02381 0 2.5 Mabsadd L closepath F
.18762 .02381 0 2.5 Mabsadd m
.18762 .02381 -0.56518 .7779 Mabsadd L
.18762 .02381 -2.37764 .77254 Mabsadd L
.18762 .02381 -0.91448 -0.29713 Mabsadd L
.18762 .02381 -1.46946 -2.02254 Mabsadd L
.18762 .02381 0 -0.96154 Mabsadd L
.18762 .02381 1.46946 -2.02254 Mabsadd L
.18762 .02381 .91448 -0.29713 Mabsadd L
.18762 .02381 2.37764 .77254 Mabsadd L
.18762 .02381 .56518 .7779 Mabsadd L
.18762 .02381 0 2.5 Mabsadd L closepath F
.18857 .02381 0 2.5 Mabsadd m
.18857 .02381 -0.56518 .7779 Mabsadd L
.18857 .02381 -2.37764 .77254 Mabsadd L
.18857 .02381 -0.91448 -0.29713 Mabsadd L
.18857 .02381 -1.46946 -2.02254 Mabsadd L
.18857 .02381 0 -0.96154 Mabsadd L
.18857 .02381 1.46946 -2.02254 Mabsadd L
.18857 .02381 .91448 -0.29713 Mabsadd L
.18857 .02381 2.37764 .77254 Mabsadd L
.18857 .02381 .56518 .7779 Mabsadd L
.18857 .02381 0 2.5 Mabsadd L closepath F
.18952 .02381 0 2.5 Mabsadd m
.18952 .02381 -0.56518 .7779 Mabsadd L
.18952 .02381 -2.37764 .77254 Mabsadd L
.18952 .02381 -0.91448 -0.29713 Mabsadd L
.18952 .02381 -1.46946 -2.02254 Mabsadd L
.18952 .02381 0 -0.96154 Mabsadd L
.18952 .02381 1.46946 -2.02254 Mabsadd L
.18952 .02381 .91448 -0.29713 Mabsadd L
.18952 .02381 2.37764 .77254 Mabsadd L
.18952 .02381 .56518 .7779 Mabsadd L
.18952 .02381 0 2.5 Mabsadd L closepath F
.19048 .02381 0 2.5 Mabsadd m
.19048 .02381 -0.56518 .7779 Mabsadd L
.19048 .02381 -2.37764 .77254 Mabsadd L
.19048 .02381 -0.91448 -0.29713 Mabsadd L
.19048 .02381 -1.46946 -2.02254 Mabsadd L
.19048 .02381 0 -0.96154 Mabsadd L
.19048 .02381 1.46946 -2.02254 Mabsadd L
.19048 .02381 .91448 -0.29713 Mabsadd L
.19048 .02381 2.37764 .77254 Mabsadd L
.19048 .02381 .56518 .7779 Mabsadd L
.19048 .02381 0 2.5 Mabsadd L closepath F
.19143 .02381 0 2.5 Mabsadd m
.19143 .02381 -0.56518 .7779 Mabsadd L
.19143 .02381 -2.37764 .77254 Mabsadd L
.19143 .02381 -0.91448 -0.29713 Mabsadd L
.19143 .02381 -1.46946 -2.02254 Mabsadd L
.19143 .02381 0 -0.96154 Mabsadd L
.19143 .02381 1.46946 -2.02254 Mabsadd L
.19143 .02381 .91448 -0.29713 Mabsadd L
.19143 .02381 2.37764 .77254 Mabsadd L
.19143 .02381 .56518 .7779 Mabsadd L
.19143 .02381 0 2.5 Mabsadd L closepath F
.19238 .02381 0 2.5 Mabsadd m
.19238 .02381 -0.56518 .7779 Mabsadd L
.19238 .02381 -2.37764 .77254 Mabsadd L
.19238 .02381 -0.91448 -0.29713 Mabsadd L
.19238 .02381 -1.46946 -2.02254 Mabsadd L
.19238 .02381 0 -0.96154 Mabsadd L
.19238 .02381 1.46946 -2.02254 Mabsadd L
.19238 .02381 .91448 -0.29713 Mabsadd L
.19238 .02381 2.37764 .77254 Mabsadd L
.19238 .02381 .56518 .7779 Mabsadd L
.19238 .02381 0 2.5 Mabsadd L closepath F
.19333 .02381 0 2.5 Mabsadd m
.19333 .02381 -0.56518 .7779 Mabsadd L
.19333 .02381 -2.37764 .77254 Mabsadd L
.19333 .02381 -0.91448 -0.29713 Mabsadd L
.19333 .02381 -1.46946 -2.02254 Mabsadd L
.19333 .02381 0 -0.96154 Mabsadd L
.19333 .02381 1.46946 -2.02254 Mabsadd L
.19333 .02381 .91448 -0.29713 Mabsadd L
.19333 .02381 2.37764 .77254 Mabsadd L
.19333 .02381 .56518 .7779 Mabsadd L
.19333 .02381 0 2.5 Mabsadd L closepath F
.19429 .02381 0 2.5 Mabsadd m
.19429 .02381 -0.56518 .7779 Mabsadd L
.19429 .02381 -2.37764 .77254 Mabsadd L
.19429 .02381 -0.91448 -0.29713 Mabsadd L
.19429 .02381 -1.46946 -2.02254 Mabsadd L
.19429 .02381 0 -0.96154 Mabsadd L
.19429 .02381 1.46946 -2.02254 Mabsadd L
.19429 .02381 .91448 -0.29713 Mabsadd L
.19429 .02381 2.37764 .77254 Mabsadd L
.19429 .02381 .56518 .7779 Mabsadd L
.19429 .02381 0 2.5 Mabsadd L closepath F
.19524 .02381 0 2.5 Mabsadd m
.19524 .02381 -0.56518 .7779 Mabsadd L
.19524 .02381 -2.37764 .77254 Mabsadd L
.19524 .02381 -0.91448 -0.29713 Mabsadd L
.19524 .02381 -1.46946 -2.02254 Mabsadd L
.19524 .02381 0 -0.96154 Mabsadd L
.19524 .02381 1.46946 -2.02254 Mabsadd L
.19524 .02381 .91448 -0.29713 Mabsadd L
.19524 .02381 2.37764 .77254 Mabsadd L
.19524 .02381 .56518 .7779 Mabsadd L
.19524 .02381 0 2.5 Mabsadd L closepath F
.19619 .02381 0 2.5 Mabsadd m
.19619 .02381 -0.56518 .7779 Mabsadd L
.19619 .02381 -2.37764 .77254 Mabsadd L
.19619 .02381 -0.91448 -0.29713 Mabsadd L
.19619 .02381 -1.46946 -2.02254 Mabsadd L
.19619 .02381 0 -0.96154 Mabsadd L
.19619 .02381 1.46946 -2.02254 Mabsadd L
.19619 .02381 .91448 -0.29713 Mabsadd L
.19619 .02381 2.37764 .77254 Mabsadd L
.19619 .02381 .56518 .7779 Mabsadd L
.19619 .02381 0 2.5 Mabsadd L closepath F
.19714 .02382 0 2.5 Mabsadd m
.19714 .02382 -0.56518 .7779 Mabsadd L
.19714 .02382 -2.37764 .77254 Mabsadd L
.19714 .02382 -0.91448 -0.29713 Mabsadd L
.19714 .02382 -1.46946 -2.02254 Mabsadd L
.19714 .02382 0 -0.96154 Mabsadd L
.19714 .02382 1.46946 -2.02254 Mabsadd L
.19714 .02382 .91448 -0.29713 Mabsadd L
.19714 .02382 2.37764 .77254 Mabsadd L
.19714 .02382 .56518 .7779 Mabsadd L
.19714 .02382 0 2.5 Mabsadd L closepath F
.1981 .02382 0 2.5 Mabsadd m
.1981 .02382 -0.56518 .7779 Mabsadd L
.1981 .02382 -2.37764 .77254 Mabsadd L
.1981 .02382 -0.91448 -0.29713 Mabsadd L
.1981 .02382 -1.46946 -2.02254 Mabsadd L
.1981 .02382 0 -0.96154 Mabsadd L
.1981 .02382 1.46946 -2.02254 Mabsadd L
.1981 .02382 .91448 -0.29713 Mabsadd L
.1981 .02382 2.37764 .77254 Mabsadd L
.1981 .02382 .56518 .7779 Mabsadd L
.1981 .02382 0 2.5 Mabsadd L closepath F
.19905 .02381 0 2.5 Mabsadd m
.19905 .02381 -0.56518 .7779 Mabsadd L
.19905 .02381 -2.37764 .77254 Mabsadd L
.19905 .02381 -0.91448 -0.29713 Mabsadd L
.19905 .02381 -1.46946 -2.02254 Mabsadd L
.19905 .02381 0 -0.96154 Mabsadd L
.19905 .02381 1.46946 -2.02254 Mabsadd L
.19905 .02381 .91448 -0.29713 Mabsadd L
.19905 .02381 2.37764 .77254 Mabsadd L
.19905 .02381 .56518 .7779 Mabsadd L
.19905 .02381 0 2.5 Mabsadd L closepath F
.2 .02382 0 2.5 Mabsadd m
.2 .02382 -0.56518 .7779 Mabsadd L
.2 .02382 -2.37764 .77254 Mabsadd L
.2 .02382 -0.91448 -0.29713 Mabsadd L
.2 .02382 -1.46946 -2.02254 Mabsadd L
.2 .02382 0 -0.96154 Mabsadd L
.2 .02382 1.46946 -2.02254 Mabsadd L
.2 .02382 .91448 -0.29713 Mabsadd L
.2 .02382 2.37764 .77254 Mabsadd L
.2 .02382 .56518 .7779 Mabsadd L
.2 .02382 0 2.5 Mabsadd L closepath F
.20095 .02382 0 2.5 Mabsadd m
.20095 .02382 -0.56518 .7779 Mabsadd L
.20095 .02382 -2.37764 .77254 Mabsadd L
.20095 .02382 -0.91448 -0.29713 Mabsadd L
.20095 .02382 -1.46946 -2.02254 Mabsadd L
.20095 .02382 0 -0.96154 Mabsadd L
.20095 .02382 1.46946 -2.02254 Mabsadd L
.20095 .02382 .91448 -0.29713 Mabsadd L
.20095 .02382 2.37764 .77254 Mabsadd L
.20095 .02382 .56518 .7779 Mabsadd L
.20095 .02382 0 2.5 Mabsadd L closepath F
.2019 .02382 0 2.5 Mabsadd m
.2019 .02382 -0.56518 .7779 Mabsadd L
.2019 .02382 -2.37764 .77254 Mabsadd L
.2019 .02382 -0.91448 -0.29713 Mabsadd L
.2019 .02382 -1.46946 -2.02254 Mabsadd L
.2019 .02382 0 -0.96154 Mabsadd L
.2019 .02382 1.46946 -2.02254 Mabsadd L
.2019 .02382 .91448 -0.29713 Mabsadd L
.2019 .02382 2.37764 .77254 Mabsadd L
.2019 .02382 .56518 .7779 Mabsadd L
.2019 .02382 0 2.5 Mabsadd L closepath F
.20286 .02382 0 2.5 Mabsadd m
.20286 .02382 -0.56518 .7779 Mabsadd L
.20286 .02382 -2.37764 .77254 Mabsadd L
.20286 .02382 -0.91448 -0.29713 Mabsadd L
.20286 .02382 -1.46946 -2.02254 Mabsadd L
.20286 .02382 0 -0.96154 Mabsadd L
.20286 .02382 1.46946 -2.02254 Mabsadd L
.20286 .02382 .91448 -0.29713 Mabsadd L
.20286 .02382 2.37764 .77254 Mabsadd L
.20286 .02382 .56518 .7779 Mabsadd L
.20286 .02382 0 2.5 Mabsadd L closepath F
.20381 .02381 0 2.5 Mabsadd m
.20381 .02381 -0.56518 .7779 Mabsadd L
.20381 .02381 -2.37764 .77254 Mabsadd L
.20381 .02381 -0.91448 -0.29713 Mabsadd L
.20381 .02381 -1.46946 -2.02254 Mabsadd L
.20381 .02381 0 -0.96154 Mabsadd L
.20381 .02381 1.46946 -2.02254 Mabsadd L
.20381 .02381 .91448 -0.29713 Mabsadd L
.20381 .02381 2.37764 .77254 Mabsadd L
.20381 .02381 .56518 .7779 Mabsadd L
.20381 .02381 0 2.5 Mabsadd L closepath F
.20476 .02382 0 2.5 Mabsadd m
.20476 .02382 -0.56518 .7779 Mabsadd L
.20476 .02382 -2.37764 .77254 Mabsadd L
.20476 .02382 -0.91448 -0.29713 Mabsadd L
.20476 .02382 -1.46946 -2.02254 Mabsadd L
.20476 .02382 0 -0.96154 Mabsadd L
.20476 .02382 1.46946 -2.02254 Mabsadd L
.20476 .02382 .91448 -0.29713 Mabsadd L
.20476 .02382 2.37764 .77254 Mabsadd L
.20476 .02382 .56518 .7779 Mabsadd L
.20476 .02382 0 2.5 Mabsadd L closepath F
.20571 .02381 0 2.5 Mabsadd m
.20571 .02381 -0.56518 .7779 Mabsadd L
.20571 .02381 -2.37764 .77254 Mabsadd L
.20571 .02381 -0.91448 -0.29713 Mabsadd L
.20571 .02381 -1.46946 -2.02254 Mabsadd L
.20571 .02381 0 -0.96154 Mabsadd L
.20571 .02381 1.46946 -2.02254 Mabsadd L
.20571 .02381 .91448 -0.29713 Mabsadd L
.20571 .02381 2.37764 .77254 Mabsadd L
.20571 .02381 .56518 .7779 Mabsadd L
.20571 .02381 0 2.5 Mabsadd L closepath F
.20667 .02381 0 2.5 Mabsadd m
.20667 .02381 -0.56518 .7779 Mabsadd L
.20667 .02381 -2.37764 .77254 Mabsadd L
.20667 .02381 -0.91448 -0.29713 Mabsadd L
.20667 .02381 -1.46946 -2.02254 Mabsadd L
.20667 .02381 0 -0.96154 Mabsadd L
.20667 .02381 1.46946 -2.02254 Mabsadd L
.20667 .02381 .91448 -0.29713 Mabsadd L
.20667 .02381 2.37764 .77254 Mabsadd L
.20667 .02381 .56518 .7779 Mabsadd L
.20667 .02381 0 2.5 Mabsadd L closepath F
.20762 .02381 0 2.5 Mabsadd m
.20762 .02381 -0.56518 .7779 Mabsadd L
.20762 .02381 -2.37764 .77254 Mabsadd L
.20762 .02381 -0.91448 -0.29713 Mabsadd L
.20762 .02381 -1.46946 -2.02254 Mabsadd L
.20762 .02381 0 -0.96154 Mabsadd L
.20762 .02381 1.46946 -2.02254 Mabsadd L
.20762 .02381 .91448 -0.29713 Mabsadd L
.20762 .02381 2.37764 .77254 Mabsadd L
.20762 .02381 .56518 .7779 Mabsadd L
.20762 .02381 0 2.5 Mabsadd L closepath F
.20857 .02381 0 2.5 Mabsadd m
.20857 .02381 -0.56518 .7779 Mabsadd L
.20857 .02381 -2.37764 .77254 Mabsadd L
.20857 .02381 -0.91448 -0.29713 Mabsadd L
.20857 .02381 -1.46946 -2.02254 Mabsadd L
.20857 .02381 0 -0.96154 Mabsadd L
.20857 .02381 1.46946 -2.02254 Mabsadd L
.20857 .02381 .91448 -0.29713 Mabsadd L
.20857 .02381 2.37764 .77254 Mabsadd L
.20857 .02381 .56518 .7779 Mabsadd L
.20857 .02381 0 2.5 Mabsadd L closepath F
.20952 .02381 0 2.5 Mabsadd m
.20952 .02381 -0.56518 .7779 Mabsadd L
.20952 .02381 -2.37764 .77254 Mabsadd L
.20952 .02381 -0.91448 -0.29713 Mabsadd L
.20952 .02381 -1.46946 -2.02254 Mabsadd L
.20952 .02381 0 -0.96154 Mabsadd L
.20952 .02381 1.46946 -2.02254 Mabsadd L
.20952 .02381 .91448 -0.29713 Mabsadd L
.20952 .02381 2.37764 .77254 Mabsadd L
.20952 .02381 .56518 .7779 Mabsadd L
.20952 .02381 0 2.5 Mabsadd L closepath F
.21048 .02382 0 2.5 Mabsadd m
.21048 .02382 -0.56518 .7779 Mabsadd L
.21048 .02382 -2.37764 .77254 Mabsadd L
.21048 .02382 -0.91448 -0.29713 Mabsadd L
.21048 .02382 -1.46946 -2.02254 Mabsadd L
.21048 .02382 0 -0.96154 Mabsadd L
.21048 .02382 1.46946 -2.02254 Mabsadd L
.21048 .02382 .91448 -0.29713 Mabsadd L
.21048 .02382 2.37764 .77254 Mabsadd L
.21048 .02382 .56518 .7779 Mabsadd L
.21048 .02382 0 2.5 Mabsadd L closepath F
.21143 .02381 0 2.5 Mabsadd m
.21143 .02381 -0.56518 .7779 Mabsadd L
.21143 .02381 -2.37764 .77254 Mabsadd L
.21143 .02381 -0.91448 -0.29713 Mabsadd L
.21143 .02381 -1.46946 -2.02254 Mabsadd L
.21143 .02381 0 -0.96154 Mabsadd L
.21143 .02381 1.46946 -2.02254 Mabsadd L
.21143 .02381 .91448 -0.29713 Mabsadd L
.21143 .02381 2.37764 .77254 Mabsadd L
.21143 .02381 .56518 .7779 Mabsadd L
.21143 .02381 0 2.5 Mabsadd L closepath F
.21238 .02381 0 2.5 Mabsadd m
.21238 .02381 -0.56518 .7779 Mabsadd L
.21238 .02381 -2.37764 .77254 Mabsadd L
.21238 .02381 -0.91448 -0.29713 Mabsadd L
.21238 .02381 -1.46946 -2.02254 Mabsadd L
.21238 .02381 0 -0.96154 Mabsadd L
.21238 .02381 1.46946 -2.02254 Mabsadd L
.21238 .02381 .91448 -0.29713 Mabsadd L
.21238 .02381 2.37764 .77254 Mabsadd L
.21238 .02381 .56518 .7779 Mabsadd L
.21238 .02381 0 2.5 Mabsadd L closepath F
.21333 .02381 0 2.5 Mabsadd m
.21333 .02381 -0.56518 .7779 Mabsadd L
.21333 .02381 -2.37764 .77254 Mabsadd L
.21333 .02381 -0.91448 -0.29713 Mabsadd L
.21333 .02381 -1.46946 -2.02254 Mabsadd L
.21333 .02381 0 -0.96154 Mabsadd L
.21333 .02381 1.46946 -2.02254 Mabsadd L
.21333 .02381 .91448 -0.29713 Mabsadd L
.21333 .02381 2.37764 .77254 Mabsadd L
.21333 .02381 .56518 .7779 Mabsadd L
.21333 .02381 0 2.5 Mabsadd L closepath F
.21429 .02381 0 2.5 Mabsadd m
.21429 .02381 -0.56518 .7779 Mabsadd L
.21429 .02381 -2.37764 .77254 Mabsadd L
.21429 .02381 -0.91448 -0.29713 Mabsadd L
.21429 .02381 -1.46946 -2.02254 Mabsadd L
.21429 .02381 0 -0.96154 Mabsadd L
.21429 .02381 1.46946 -2.02254 Mabsadd L
.21429 .02381 .91448 -0.29713 Mabsadd L
.21429 .02381 2.37764 .77254 Mabsadd L
.21429 .02381 .56518 .7779 Mabsadd L
.21429 .02381 0 2.5 Mabsadd L closepath F
.21524 .02381 0 2.5 Mabsadd m
.21524 .02381 -0.56518 .7779 Mabsadd L
.21524 .02381 -2.37764 .77254 Mabsadd L
.21524 .02381 -0.91448 -0.29713 Mabsadd L
.21524 .02381 -1.46946 -2.02254 Mabsadd L
.21524 .02381 0 -0.96154 Mabsadd L
.21524 .02381 1.46946 -2.02254 Mabsadd L
.21524 .02381 .91448 -0.29713 Mabsadd L
.21524 .02381 2.37764 .77254 Mabsadd L
.21524 .02381 .56518 .7779 Mabsadd L
.21524 .02381 0 2.5 Mabsadd L closepath F
.21619 .02382 0 2.5 Mabsadd m
.21619 .02382 -0.56518 .7779 Mabsadd L
.21619 .02382 -2.37764 .77254 Mabsadd L
.21619 .02382 -0.91448 -0.29713 Mabsadd L
.21619 .02382 -1.46946 -2.02254 Mabsadd L
.21619 .02382 0 -0.96154 Mabsadd L
.21619 .02382 1.46946 -2.02254 Mabsadd L
.21619 .02382 .91448 -0.29713 Mabsadd L
.21619 .02382 2.37764 .77254 Mabsadd L
.21619 .02382 .56518 .7779 Mabsadd L
.21619 .02382 0 2.5 Mabsadd L closepath F
.21714 .02381 0 2.5 Mabsadd m
.21714 .02381 -0.56518 .7779 Mabsadd L
.21714 .02381 -2.37764 .77254 Mabsadd L
.21714 .02381 -0.91448 -0.29713 Mabsadd L
.21714 .02381 -1.46946 -2.02254 Mabsadd L
.21714 .02381 0 -0.96154 Mabsadd L
.21714 .02381 1.46946 -2.02254 Mabsadd L
.21714 .02381 .91448 -0.29713 Mabsadd L
.21714 .02381 2.37764 .77254 Mabsadd L
.21714 .02381 .56518 .7779 Mabsadd L
.21714 .02381 0 2.5 Mabsadd L closepath F
.2181 .02382 0 2.5 Mabsadd m
.2181 .02382 -0.56518 .7779 Mabsadd L
.2181 .02382 -2.37764 .77254 Mabsadd L
.2181 .02382 -0.91448 -0.29713 Mabsadd L
.2181 .02382 -1.46946 -2.02254 Mabsadd L
.2181 .02382 0 -0.96154 Mabsadd L
.2181 .02382 1.46946 -2.02254 Mabsadd L
.2181 .02382 .91448 -0.29713 Mabsadd L
.2181 .02382 2.37764 .77254 Mabsadd L
.2181 .02382 .56518 .7779 Mabsadd L
.2181 .02382 0 2.5 Mabsadd L closepath F
.21905 .02382 0 2.5 Mabsadd m
.21905 .02382 -0.56518 .7779 Mabsadd L
.21905 .02382 -2.37764 .77254 Mabsadd L
.21905 .02382 -0.91448 -0.29713 Mabsadd L
.21905 .02382 -1.46946 -2.02254 Mabsadd L
.21905 .02382 0 -0.96154 Mabsadd L
.21905 .02382 1.46946 -2.02254 Mabsadd L
.21905 .02382 .91448 -0.29713 Mabsadd L
.21905 .02382 2.37764 .77254 Mabsadd L
.21905 .02382 .56518 .7779 Mabsadd L
.21905 .02382 0 2.5 Mabsadd L closepath F
.22 .02382 0 2.5 Mabsadd m
.22 .02382 -0.56518 .7779 Mabsadd L
.22 .02382 -2.37764 .77254 Mabsadd L
.22 .02382 -0.91448 -0.29713 Mabsadd L
.22 .02382 -1.46946 -2.02254 Mabsadd L
.22 .02382 0 -0.96154 Mabsadd L
.22 .02382 1.46946 -2.02254 Mabsadd L
.22 .02382 .91448 -0.29713 Mabsadd L
.22 .02382 2.37764 .77254 Mabsadd L
.22 .02382 .56518 .7779 Mabsadd L
.22 .02382 0 2.5 Mabsadd L closepath F
.22095 .02382 0 2.5 Mabsadd m
.22095 .02382 -0.56518 .7779 Mabsadd L
.22095 .02382 -2.37764 .77254 Mabsadd L
.22095 .02382 -0.91448 -0.29713 Mabsadd L
.22095 .02382 -1.46946 -2.02254 Mabsadd L
.22095 .02382 0 -0.96154 Mabsadd L
.22095 .02382 1.46946 -2.02254 Mabsadd L
.22095 .02382 .91448 -0.29713 Mabsadd L
.22095 .02382 2.37764 .77254 Mabsadd L
.22095 .02382 .56518 .7779 Mabsadd L
.22095 .02382 0 2.5 Mabsadd L closepath F
.2219 .02382 0 2.5 Mabsadd m
.2219 .02382 -0.56518 .7779 Mabsadd L
.2219 .02382 -2.37764 .77254 Mabsadd L
.2219 .02382 -0.91448 -0.29713 Mabsadd L
.2219 .02382 -1.46946 -2.02254 Mabsadd L
.2219 .02382 0 -0.96154 Mabsadd L
.2219 .02382 1.46946 -2.02254 Mabsadd L
.2219 .02382 .91448 -0.29713 Mabsadd L
.2219 .02382 2.37764 .77254 Mabsadd L
.2219 .02382 .56518 .7779 Mabsadd L
.2219 .02382 0 2.5 Mabsadd L closepath F
.22286 .02382 0 2.5 Mabsadd m
.22286 .02382 -0.56518 .7779 Mabsadd L
.22286 .02382 -2.37764 .77254 Mabsadd L
.22286 .02382 -0.91448 -0.29713 Mabsadd L
.22286 .02382 -1.46946 -2.02254 Mabsadd L
.22286 .02382 0 -0.96154 Mabsadd L
.22286 .02382 1.46946 -2.02254 Mabsadd L
.22286 .02382 .91448 -0.29713 Mabsadd L
.22286 .02382 2.37764 .77254 Mabsadd L
.22286 .02382 .56518 .7779 Mabsadd L
.22286 .02382 0 2.5 Mabsadd L closepath F
.22381 .02382 0 2.5 Mabsadd m
.22381 .02382 -0.56518 .7779 Mabsadd L
.22381 .02382 -2.37764 .77254 Mabsadd L
.22381 .02382 -0.91448 -0.29713 Mabsadd L
.22381 .02382 -1.46946 -2.02254 Mabsadd L
.22381 .02382 0 -0.96154 Mabsadd L
.22381 .02382 1.46946 -2.02254 Mabsadd L
.22381 .02382 .91448 -0.29713 Mabsadd L
.22381 .02382 2.37764 .77254 Mabsadd L
.22381 .02382 .56518 .7779 Mabsadd L
.22381 .02382 0 2.5 Mabsadd L closepath F
.22476 .02382 0 2.5 Mabsadd m
.22476 .02382 -0.56518 .7779 Mabsadd L
.22476 .02382 -2.37764 .77254 Mabsadd L
.22476 .02382 -0.91448 -0.29713 Mabsadd L
.22476 .02382 -1.46946 -2.02254 Mabsadd L
.22476 .02382 0 -0.96154 Mabsadd L
.22476 .02382 1.46946 -2.02254 Mabsadd L
.22476 .02382 .91448 -0.29713 Mabsadd L
.22476 .02382 2.37764 .77254 Mabsadd L
.22476 .02382 .56518 .7779 Mabsadd L
.22476 .02382 0 2.5 Mabsadd L closepath F
.22571 .02382 0 2.5 Mabsadd m
.22571 .02382 -0.56518 .7779 Mabsadd L
.22571 .02382 -2.37764 .77254 Mabsadd L
.22571 .02382 -0.91448 -0.29713 Mabsadd L
.22571 .02382 -1.46946 -2.02254 Mabsadd L
.22571 .02382 0 -0.96154 Mabsadd L
.22571 .02382 1.46946 -2.02254 Mabsadd L
.22571 .02382 .91448 -0.29713 Mabsadd L
.22571 .02382 2.37764 .77254 Mabsadd L
.22571 .02382 .56518 .7779 Mabsadd L
.22571 .02382 0 2.5 Mabsadd L closepath F
.22667 .02382 0 2.5 Mabsadd m
.22667 .02382 -0.56518 .7779 Mabsadd L
.22667 .02382 -2.37764 .77254 Mabsadd L
.22667 .02382 -0.91448 -0.29713 Mabsadd L
.22667 .02382 -1.46946 -2.02254 Mabsadd L
.22667 .02382 0 -0.96154 Mabsadd L
.22667 .02382 1.46946 -2.02254 Mabsadd L
.22667 .02382 .91448 -0.29713 Mabsadd L
.22667 .02382 2.37764 .77254 Mabsadd L
.22667 .02382 .56518 .7779 Mabsadd L
.22667 .02382 0 2.5 Mabsadd L closepath F
.22762 .02382 0 2.5 Mabsadd m
.22762 .02382 -0.56518 .7779 Mabsadd L
.22762 .02382 -2.37764 .77254 Mabsadd L
.22762 .02382 -0.91448 -0.29713 Mabsadd L
.22762 .02382 -1.46946 -2.02254 Mabsadd L
.22762 .02382 0 -0.96154 Mabsadd L
.22762 .02382 1.46946 -2.02254 Mabsadd L
.22762 .02382 .91448 -0.29713 Mabsadd L
.22762 .02382 2.37764 .77254 Mabsadd L
.22762 .02382 .56518 .7779 Mabsadd L
.22762 .02382 0 2.5 Mabsadd L closepath F
.22857 .02382 0 2.5 Mabsadd m
.22857 .02382 -0.56518 .7779 Mabsadd L
.22857 .02382 -2.37764 .77254 Mabsadd L
.22857 .02382 -0.91448 -0.29713 Mabsadd L
.22857 .02382 -1.46946 -2.02254 Mabsadd L
.22857 .02382 0 -0.96154 Mabsadd L
.22857 .02382 1.46946 -2.02254 Mabsadd L
.22857 .02382 .91448 -0.29713 Mabsadd L
.22857 .02382 2.37764 .77254 Mabsadd L
.22857 .02382 .56518 .7779 Mabsadd L
.22857 .02382 0 2.5 Mabsadd L closepath F
.22952 .02382 0 2.5 Mabsadd m
.22952 .02382 -0.56518 .7779 Mabsadd L
.22952 .02382 -2.37764 .77254 Mabsadd L
.22952 .02382 -0.91448 -0.29713 Mabsadd L
.22952 .02382 -1.46946 -2.02254 Mabsadd L
.22952 .02382 0 -0.96154 Mabsadd L
.22952 .02382 1.46946 -2.02254 Mabsadd L
.22952 .02382 .91448 -0.29713 Mabsadd L
.22952 .02382 2.37764 .77254 Mabsadd L
.22952 .02382 .56518 .7779 Mabsadd L
.22952 .02382 0 2.5 Mabsadd L closepath F
.23048 .02382 0 2.5 Mabsadd m
.23048 .02382 -0.56518 .7779 Mabsadd L
.23048 .02382 -2.37764 .77254 Mabsadd L
.23048 .02382 -0.91448 -0.29713 Mabsadd L
.23048 .02382 -1.46946 -2.02254 Mabsadd L
.23048 .02382 0 -0.96154 Mabsadd L
.23048 .02382 1.46946 -2.02254 Mabsadd L
.23048 .02382 .91448 -0.29713 Mabsadd L
.23048 .02382 2.37764 .77254 Mabsadd L
.23048 .02382 .56518 .7779 Mabsadd L
.23048 .02382 0 2.5 Mabsadd L closepath F
.23143 .02382 0 2.5 Mabsadd m
.23143 .02382 -0.56518 .7779 Mabsadd L
.23143 .02382 -2.37764 .77254 Mabsadd L
.23143 .02382 -0.91448 -0.29713 Mabsadd L
.23143 .02382 -1.46946 -2.02254 Mabsadd L
.23143 .02382 0 -0.96154 Mabsadd L
.23143 .02382 1.46946 -2.02254 Mabsadd L
.23143 .02382 .91448 -0.29713 Mabsadd L
.23143 .02382 2.37764 .77254 Mabsadd L
.23143 .02382 .56518 .7779 Mabsadd L
.23143 .02382 0 2.5 Mabsadd L closepath F
.23238 .02382 0 2.5 Mabsadd m
.23238 .02382 -0.56518 .7779 Mabsadd L
.23238 .02382 -2.37764 .77254 Mabsadd L
.23238 .02382 -0.91448 -0.29713 Mabsadd L
.23238 .02382 -1.46946 -2.02254 Mabsadd L
.23238 .02382 0 -0.96154 Mabsadd L
.23238 .02382 1.46946 -2.02254 Mabsadd L
.23238 .02382 .91448 -0.29713 Mabsadd L
.23238 .02382 2.37764 .77254 Mabsadd L
.23238 .02382 .56518 .7779 Mabsadd L
.23238 .02382 0 2.5 Mabsadd L closepath F
.23333 .02382 0 2.5 Mabsadd m
.23333 .02382 -0.56518 .7779 Mabsadd L
.23333 .02382 -2.37764 .77254 Mabsadd L
.23333 .02382 -0.91448 -0.29713 Mabsadd L
.23333 .02382 -1.46946 -2.02254 Mabsadd L
.23333 .02382 0 -0.96154 Mabsadd L
.23333 .02382 1.46946 -2.02254 Mabsadd L
.23333 .02382 .91448 -0.29713 Mabsadd L
.23333 .02382 2.37764 .77254 Mabsadd L
.23333 .02382 .56518 .7779 Mabsadd L
.23333 .02382 0 2.5 Mabsadd L closepath F
.23429 .02382 0 2.5 Mabsadd m
.23429 .02382 -0.56518 .7779 Mabsadd L
.23429 .02382 -2.37764 .77254 Mabsadd L
.23429 .02382 -0.91448 -0.29713 Mabsadd L
.23429 .02382 -1.46946 -2.02254 Mabsadd L
.23429 .02382 0 -0.96154 Mabsadd L
.23429 .02382 1.46946 -2.02254 Mabsadd L
.23429 .02382 .91448 -0.29713 Mabsadd L
.23429 .02382 2.37764 .77254 Mabsadd L
.23429 .02382 .56518 .7779 Mabsadd L
.23429 .02382 0 2.5 Mabsadd L closepath F
.23524 .02382 0 2.5 Mabsadd m
.23524 .02382 -0.56518 .7779 Mabsadd L
.23524 .02382 -2.37764 .77254 Mabsadd L
.23524 .02382 -0.91448 -0.29713 Mabsadd L
.23524 .02382 -1.46946 -2.02254 Mabsadd L
.23524 .02382 0 -0.96154 Mabsadd L
.23524 .02382 1.46946 -2.02254 Mabsadd L
.23524 .02382 .91448 -0.29713 Mabsadd L
.23524 .02382 2.37764 .77254 Mabsadd L
.23524 .02382 .56518 .7779 Mabsadd L
.23524 .02382 0 2.5 Mabsadd L closepath F
.23619 .02382 0 2.5 Mabsadd m
.23619 .02382 -0.56518 .7779 Mabsadd L
.23619 .02382 -2.37764 .77254 Mabsadd L
.23619 .02382 -0.91448 -0.29713 Mabsadd L
.23619 .02382 -1.46946 -2.02254 Mabsadd L
.23619 .02382 0 -0.96154 Mabsadd L
.23619 .02382 1.46946 -2.02254 Mabsadd L
.23619 .02382 .91448 -0.29713 Mabsadd L
.23619 .02382 2.37764 .77254 Mabsadd L
.23619 .02382 .56518 .7779 Mabsadd L
.23619 .02382 0 2.5 Mabsadd L closepath F
.23714 .02382 0 2.5 Mabsadd m
.23714 .02382 -0.56518 .7779 Mabsadd L
.23714 .02382 -2.37764 .77254 Mabsadd L
.23714 .02382 -0.91448 -0.29713 Mabsadd L
.23714 .02382 -1.46946 -2.02254 Mabsadd L
.23714 .02382 0 -0.96154 Mabsadd L
.23714 .02382 1.46946 -2.02254 Mabsadd L
.23714 .02382 .91448 -0.29713 Mabsadd L
.23714 .02382 2.37764 .77254 Mabsadd L
.23714 .02382 .56518 .7779 Mabsadd L
.23714 .02382 0 2.5 Mabsadd L closepath F
.2381 .02382 0 2.5 Mabsadd m
.2381 .02382 -0.56518 .7779 Mabsadd L
.2381 .02382 -2.37764 .77254 Mabsadd L
.2381 .02382 -0.91448 -0.29713 Mabsadd L
.2381 .02382 -1.46946 -2.02254 Mabsadd L
.2381 .02382 0 -0.96154 Mabsadd L
.2381 .02382 1.46946 -2.02254 Mabsadd L
.2381 .02382 .91448 -0.29713 Mabsadd L
.2381 .02382 2.37764 .77254 Mabsadd L
.2381 .02382 .56518 .7779 Mabsadd L
.2381 .02382 0 2.5 Mabsadd L closepath F
.23905 .02382 0 2.5 Mabsadd m
.23905 .02382 -0.56518 .7779 Mabsadd L
.23905 .02382 -2.37764 .77254 Mabsadd L
.23905 .02382 -0.91448 -0.29713 Mabsadd L
.23905 .02382 -1.46946 -2.02254 Mabsadd L
.23905 .02382 0 -0.96154 Mabsadd L
.23905 .02382 1.46946 -2.02254 Mabsadd L
.23905 .02382 .91448 -0.29713 Mabsadd L
.23905 .02382 2.37764 .77254 Mabsadd L
.23905 .02382 .56518 .7779 Mabsadd L
.23905 .02382 0 2.5 Mabsadd L closepath F
.24 .02382 0 2.5 Mabsadd m
.24 .02382 -0.56518 .7779 Mabsadd L
.24 .02382 -2.37764 .77254 Mabsadd L
.24 .02382 -0.91448 -0.29713 Mabsadd L
.24 .02382 -1.46946 -2.02254 Mabsadd L
.24 .02382 0 -0.96154 Mabsadd L
.24 .02382 1.46946 -2.02254 Mabsadd L
.24 .02382 .91448 -0.29713 Mabsadd L
.24 .02382 2.37764 .77254 Mabsadd L
.24 .02382 .56518 .7779 Mabsadd L
.24 .02382 0 2.5 Mabsadd L closepath F
.24095 .02382 0 2.5 Mabsadd m
.24095 .02382 -0.56518 .7779 Mabsadd L
.24095 .02382 -2.37764 .77254 Mabsadd L
.24095 .02382 -0.91448 -0.29713 Mabsadd L
.24095 .02382 -1.46946 -2.02254 Mabsadd L
.24095 .02382 0 -0.96154 Mabsadd L
.24095 .02382 1.46946 -2.02254 Mabsadd L
.24095 .02382 .91448 -0.29713 Mabsadd L
.24095 .02382 2.37764 .77254 Mabsadd L
.24095 .02382 .56518 .7779 Mabsadd L
.24095 .02382 0 2.5 Mabsadd L closepath F
.2419 .02381 0 2.5 Mabsadd m
.2419 .02381 -0.56518 .7779 Mabsadd L
.2419 .02381 -2.37764 .77254 Mabsadd L
.2419 .02381 -0.91448 -0.29713 Mabsadd L
.2419 .02381 -1.46946 -2.02254 Mabsadd L
.2419 .02381 0 -0.96154 Mabsadd L
.2419 .02381 1.46946 -2.02254 Mabsadd L
.2419 .02381 .91448 -0.29713 Mabsadd L
.2419 .02381 2.37764 .77254 Mabsadd L
.2419 .02381 .56518 .7779 Mabsadd L
.2419 .02381 0 2.5 Mabsadd L closepath F
.24286 .02382 0 2.5 Mabsadd m
.24286 .02382 -0.56518 .7779 Mabsadd L
.24286 .02382 -2.37764 .77254 Mabsadd L
.24286 .02382 -0.91448 -0.29713 Mabsadd L
.24286 .02382 -1.46946 -2.02254 Mabsadd L
.24286 .02382 0 -0.96154 Mabsadd L
.24286 .02382 1.46946 -2.02254 Mabsadd L
.24286 .02382 .91448 -0.29713 Mabsadd L
.24286 .02382 2.37764 .77254 Mabsadd L
.24286 .02382 .56518 .7779 Mabsadd L
.24286 .02382 0 2.5 Mabsadd L closepath F
.24381 .02382 0 2.5 Mabsadd m
.24381 .02382 -0.56518 .7779 Mabsadd L
.24381 .02382 -2.37764 .77254 Mabsadd L
.24381 .02382 -0.91448 -0.29713 Mabsadd L
.24381 .02382 -1.46946 -2.02254 Mabsadd L
.24381 .02382 0 -0.96154 Mabsadd L
.24381 .02382 1.46946 -2.02254 Mabsadd L
.24381 .02382 .91448 -0.29713 Mabsadd L
.24381 .02382 2.37764 .77254 Mabsadd L
.24381 .02382 .56518 .7779 Mabsadd L
.24381 .02382 0 2.5 Mabsadd L closepath F
.24476 .02382 0 2.5 Mabsadd m
.24476 .02382 -0.56518 .7779 Mabsadd L
.24476 .02382 -2.37764 .77254 Mabsadd L
.24476 .02382 -0.91448 -0.29713 Mabsadd L
.24476 .02382 -1.46946 -2.02254 Mabsadd L
.24476 .02382 0 -0.96154 Mabsadd L
.24476 .02382 1.46946 -2.02254 Mabsadd L
.24476 .02382 .91448 -0.29713 Mabsadd L
.24476 .02382 2.37764 .77254 Mabsadd L
.24476 .02382 .56518 .7779 Mabsadd L
.24476 .02382 0 2.5 Mabsadd L closepath F
.24571 .02382 0 2.5 Mabsadd m
.24571 .02382 -0.56518 .7779 Mabsadd L
.24571 .02382 -2.37764 .77254 Mabsadd L
.24571 .02382 -0.91448 -0.29713 Mabsadd L
.24571 .02382 -1.46946 -2.02254 Mabsadd L
.24571 .02382 0 -0.96154 Mabsadd L
.24571 .02382 1.46946 -2.02254 Mabsadd L
.24571 .02382 .91448 -0.29713 Mabsadd L
.24571 .02382 2.37764 .77254 Mabsadd L
.24571 .02382 .56518 .7779 Mabsadd L
.24571 .02382 0 2.5 Mabsadd L closepath F
.24667 .02382 0 2.5 Mabsadd m
.24667 .02382 -0.56518 .7779 Mabsadd L
.24667 .02382 -2.37764 .77254 Mabsadd L
.24667 .02382 -0.91448 -0.29713 Mabsadd L
.24667 .02382 -1.46946 -2.02254 Mabsadd L
.24667 .02382 0 -0.96154 Mabsadd L
.24667 .02382 1.46946 -2.02254 Mabsadd L
.24667 .02382 .91448 -0.29713 Mabsadd L
.24667 .02382 2.37764 .77254 Mabsadd L
.24667 .02382 .56518 .7779 Mabsadd L
.24667 .02382 0 2.5 Mabsadd L closepath F
.24762 .02382 0 2.5 Mabsadd m
.24762 .02382 -0.56518 .7779 Mabsadd L
.24762 .02382 -2.37764 .77254 Mabsadd L
.24762 .02382 -0.91448 -0.29713 Mabsadd L
.24762 .02382 -1.46946 -2.02254 Mabsadd L
.24762 .02382 0 -0.96154 Mabsadd L
.24762 .02382 1.46946 -2.02254 Mabsadd L
.24762 .02382 .91448 -0.29713 Mabsadd L
.24762 .02382 2.37764 .77254 Mabsadd L
.24762 .02382 .56518 .7779 Mabsadd L
.24762 .02382 0 2.5 Mabsadd L closepath F
.24857 .02382 0 2.5 Mabsadd m
.24857 .02382 -0.56518 .7779 Mabsadd L
.24857 .02382 -2.37764 .77254 Mabsadd L
.24857 .02382 -0.91448 -0.29713 Mabsadd L
.24857 .02382 -1.46946 -2.02254 Mabsadd L
.24857 .02382 0 -0.96154 Mabsadd L
.24857 .02382 1.46946 -2.02254 Mabsadd L
.24857 .02382 .91448 -0.29713 Mabsadd L
.24857 .02382 2.37764 .77254 Mabsadd L
.24857 .02382 .56518 .7779 Mabsadd L
.24857 .02382 0 2.5 Mabsadd L closepath F
.24952 .02382 0 2.5 Mabsadd m
.24952 .02382 -0.56518 .7779 Mabsadd L
.24952 .02382 -2.37764 .77254 Mabsadd L
.24952 .02382 -0.91448 -0.29713 Mabsadd L
.24952 .02382 -1.46946 -2.02254 Mabsadd L
.24952 .02382 0 -0.96154 Mabsadd L
.24952 .02382 1.46946 -2.02254 Mabsadd L
.24952 .02382 .91448 -0.29713 Mabsadd L
.24952 .02382 2.37764 .77254 Mabsadd L
.24952 .02382 .56518 .7779 Mabsadd L
.24952 .02382 0 2.5 Mabsadd L closepath F
.25048 .02382 0 2.5 Mabsadd m
.25048 .02382 -0.56518 .7779 Mabsadd L
.25048 .02382 -2.37764 .77254 Mabsadd L
.25048 .02382 -0.91448 -0.29713 Mabsadd L
.25048 .02382 -1.46946 -2.02254 Mabsadd L
.25048 .02382 0 -0.96154 Mabsadd L
.25048 .02382 1.46946 -2.02254 Mabsadd L
.25048 .02382 .91448 -0.29713 Mabsadd L
.25048 .02382 2.37764 .77254 Mabsadd L
.25048 .02382 .56518 .7779 Mabsadd L
.25048 .02382 0 2.5 Mabsadd L closepath F
.25143 .02382 0 2.5 Mabsadd m
.25143 .02382 -0.56518 .7779 Mabsadd L
.25143 .02382 -2.37764 .77254 Mabsadd L
.25143 .02382 -0.91448 -0.29713 Mabsadd L
.25143 .02382 -1.46946 -2.02254 Mabsadd L
.25143 .02382 0 -0.96154 Mabsadd L
.25143 .02382 1.46946 -2.02254 Mabsadd L
.25143 .02382 .91448 -0.29713 Mabsadd L
.25143 .02382 2.37764 .77254 Mabsadd L
.25143 .02382 .56518 .7779 Mabsadd L
.25143 .02382 0 2.5 Mabsadd L closepath F
.25238 .02382 0 2.5 Mabsadd m
.25238 .02382 -0.56518 .7779 Mabsadd L
.25238 .02382 -2.37764 .77254 Mabsadd L
.25238 .02382 -0.91448 -0.29713 Mabsadd L
.25238 .02382 -1.46946 -2.02254 Mabsadd L
.25238 .02382 0 -0.96154 Mabsadd L
.25238 .02382 1.46946 -2.02254 Mabsadd L
.25238 .02382 .91448 -0.29713 Mabsadd L
.25238 .02382 2.37764 .77254 Mabsadd L
.25238 .02382 .56518 .7779 Mabsadd L
.25238 .02382 0 2.5 Mabsadd L closepath F
.25333 .02382 0 2.5 Mabsadd m
.25333 .02382 -0.56518 .7779 Mabsadd L
.25333 .02382 -2.37764 .77254 Mabsadd L
.25333 .02382 -0.91448 -0.29713 Mabsadd L
.25333 .02382 -1.46946 -2.02254 Mabsadd L
.25333 .02382 0 -0.96154 Mabsadd L
.25333 .02382 1.46946 -2.02254 Mabsadd L
.25333 .02382 .91448 -0.29713 Mabsadd L
.25333 .02382 2.37764 .77254 Mabsadd L
.25333 .02382 .56518 .7779 Mabsadd L
.25333 .02382 0 2.5 Mabsadd L closepath F
.25429 .02382 0 2.5 Mabsadd m
.25429 .02382 -0.56518 .7779 Mabsadd L
.25429 .02382 -2.37764 .77254 Mabsadd L
.25429 .02382 -0.91448 -0.29713 Mabsadd L
.25429 .02382 -1.46946 -2.02254 Mabsadd L
.25429 .02382 0 -0.96154 Mabsadd L
.25429 .02382 1.46946 -2.02254 Mabsadd L
.25429 .02382 .91448 -0.29713 Mabsadd L
.25429 .02382 2.37764 .77254 Mabsadd L
.25429 .02382 .56518 .7779 Mabsadd L
.25429 .02382 0 2.5 Mabsadd L closepath F
.25524 .02382 0 2.5 Mabsadd m
.25524 .02382 -0.56518 .7779 Mabsadd L
.25524 .02382 -2.37764 .77254 Mabsadd L
.25524 .02382 -0.91448 -0.29713 Mabsadd L
.25524 .02382 -1.46946 -2.02254 Mabsadd L
.25524 .02382 0 -0.96154 Mabsadd L
.25524 .02382 1.46946 -2.02254 Mabsadd L
.25524 .02382 .91448 -0.29713 Mabsadd L
.25524 .02382 2.37764 .77254 Mabsadd L
.25524 .02382 .56518 .7779 Mabsadd L
.25524 .02382 0 2.5 Mabsadd L closepath F
.25619 .02382 0 2.5 Mabsadd m
.25619 .02382 -0.56518 .7779 Mabsadd L
.25619 .02382 -2.37764 .77254 Mabsadd L
.25619 .02382 -0.91448 -0.29713 Mabsadd L
.25619 .02382 -1.46946 -2.02254 Mabsadd L
.25619 .02382 0 -0.96154 Mabsadd L
.25619 .02382 1.46946 -2.02254 Mabsadd L
.25619 .02382 .91448 -0.29713 Mabsadd L
.25619 .02382 2.37764 .77254 Mabsadd L
.25619 .02382 .56518 .7779 Mabsadd L
.25619 .02382 0 2.5 Mabsadd L closepath F
.25714 .02382 0 2.5 Mabsadd m
.25714 .02382 -0.56518 .7779 Mabsadd L
.25714 .02382 -2.37764 .77254 Mabsadd L
.25714 .02382 -0.91448 -0.29713 Mabsadd L
.25714 .02382 -1.46946 -2.02254 Mabsadd L
.25714 .02382 0 -0.96154 Mabsadd L
.25714 .02382 1.46946 -2.02254 Mabsadd L
.25714 .02382 .91448 -0.29713 Mabsadd L
.25714 .02382 2.37764 .77254 Mabsadd L
.25714 .02382 .56518 .7779 Mabsadd L
.25714 .02382 0 2.5 Mabsadd L closepath F
.2581 .02382 0 2.5 Mabsadd m
.2581 .02382 -0.56518 .7779 Mabsadd L
.2581 .02382 -2.37764 .77254 Mabsadd L
.2581 .02382 -0.91448 -0.29713 Mabsadd L
.2581 .02382 -1.46946 -2.02254 Mabsadd L
.2581 .02382 0 -0.96154 Mabsadd L
.2581 .02382 1.46946 -2.02254 Mabsadd L
.2581 .02382 .91448 -0.29713 Mabsadd L
.2581 .02382 2.37764 .77254 Mabsadd L
.2581 .02382 .56518 .7779 Mabsadd L
.2581 .02382 0 2.5 Mabsadd L closepath F
.25905 .02382 0 2.5 Mabsadd m
.25905 .02382 -0.56518 .7779 Mabsadd L
.25905 .02382 -2.37764 .77254 Mabsadd L
.25905 .02382 -0.91448 -0.29713 Mabsadd L
.25905 .02382 -1.46946 -2.02254 Mabsadd L
.25905 .02382 0 -0.96154 Mabsadd L
.25905 .02382 1.46946 -2.02254 Mabsadd L
.25905 .02382 .91448 -0.29713 Mabsadd L
.25905 .02382 2.37764 .77254 Mabsadd L
.25905 .02382 .56518 .7779 Mabsadd L
.25905 .02382 0 2.5 Mabsadd L closepath F
.26 .02382 0 2.5 Mabsadd m
.26 .02382 -0.56518 .7779 Mabsadd L
.26 .02382 -2.37764 .77254 Mabsadd L
.26 .02382 -0.91448 -0.29713 Mabsadd L
.26 .02382 -1.46946 -2.02254 Mabsadd L
.26 .02382 0 -0.96154 Mabsadd L
.26 .02382 1.46946 -2.02254 Mabsadd L
.26 .02382 .91448 -0.29713 Mabsadd L
.26 .02382 2.37764 .77254 Mabsadd L
.26 .02382 .56518 .7779 Mabsadd L
.26 .02382 0 2.5 Mabsadd L closepath F
.26095 .02382 0 2.5 Mabsadd m
.26095 .02382 -0.56518 .7779 Mabsadd L
.26095 .02382 -2.37764 .77254 Mabsadd L
.26095 .02382 -0.91448 -0.29713 Mabsadd L
.26095 .02382 -1.46946 -2.02254 Mabsadd L
.26095 .02382 0 -0.96154 Mabsadd L
.26095 .02382 1.46946 -2.02254 Mabsadd L
.26095 .02382 .91448 -0.29713 Mabsadd L
.26095 .02382 2.37764 .77254 Mabsadd L
.26095 .02382 .56518 .7779 Mabsadd L
.26095 .02382 0 2.5 Mabsadd L closepath F
.2619 .02382 0 2.5 Mabsadd m
.2619 .02382 -0.56518 .7779 Mabsadd L
.2619 .02382 -2.37764 .77254 Mabsadd L
.2619 .02382 -0.91448 -0.29713 Mabsadd L
.2619 .02382 -1.46946 -2.02254 Mabsadd L
.2619 .02382 0 -0.96154 Mabsadd L
.2619 .02382 1.46946 -2.02254 Mabsadd L
.2619 .02382 .91448 -0.29713 Mabsadd L
.2619 .02382 2.37764 .77254 Mabsadd L
.2619 .02382 .56518 .7779 Mabsadd L
.2619 .02382 0 2.5 Mabsadd L closepath F
.26286 .02382 0 2.5 Mabsadd m
.26286 .02382 -0.56518 .7779 Mabsadd L
.26286 .02382 -2.37764 .77254 Mabsadd L
.26286 .02382 -0.91448 -0.29713 Mabsadd L
.26286 .02382 -1.46946 -2.02254 Mabsadd L
.26286 .02382 0 -0.96154 Mabsadd L
.26286 .02382 1.46946 -2.02254 Mabsadd L
.26286 .02382 .91448 -0.29713 Mabsadd L
.26286 .02382 2.37764 .77254 Mabsadd L
.26286 .02382 .56518 .7779 Mabsadd L
.26286 .02382 0 2.5 Mabsadd L closepath F
.26381 .02382 0 2.5 Mabsadd m
.26381 .02382 -0.56518 .7779 Mabsadd L
.26381 .02382 -2.37764 .77254 Mabsadd L
.26381 .02382 -0.91448 -0.29713 Mabsadd L
.26381 .02382 -1.46946 -2.02254 Mabsadd L
.26381 .02382 0 -0.96154 Mabsadd L
.26381 .02382 1.46946 -2.02254 Mabsadd L
.26381 .02382 .91448 -0.29713 Mabsadd L
.26381 .02382 2.37764 .77254 Mabsadd L
.26381 .02382 .56518 .7779 Mabsadd L
.26381 .02382 0 2.5 Mabsadd L closepath F
.26476 .02382 0 2.5 Mabsadd m
.26476 .02382 -0.56518 .7779 Mabsadd L
.26476 .02382 -2.37764 .77254 Mabsadd L
.26476 .02382 -0.91448 -0.29713 Mabsadd L
.26476 .02382 -1.46946 -2.02254 Mabsadd L
.26476 .02382 0 -0.96154 Mabsadd L
.26476 .02382 1.46946 -2.02254 Mabsadd L
.26476 .02382 .91448 -0.29713 Mabsadd L
.26476 .02382 2.37764 .77254 Mabsadd L
.26476 .02382 .56518 .7779 Mabsadd L
.26476 .02382 0 2.5 Mabsadd L closepath F
.26571 .02382 0 2.5 Mabsadd m
.26571 .02382 -0.56518 .7779 Mabsadd L
.26571 .02382 -2.37764 .77254 Mabsadd L
.26571 .02382 -0.91448 -0.29713 Mabsadd L
.26571 .02382 -1.46946 -2.02254 Mabsadd L
.26571 .02382 0 -0.96154 Mabsadd L
.26571 .02382 1.46946 -2.02254 Mabsadd L
.26571 .02382 .91448 -0.29713 Mabsadd L
.26571 .02382 2.37764 .77254 Mabsadd L
.26571 .02382 .56518 .7779 Mabsadd L
.26571 .02382 0 2.5 Mabsadd L closepath F
.26667 .02382 0 2.5 Mabsadd m
.26667 .02382 -0.56518 .7779 Mabsadd L
.26667 .02382 -2.37764 .77254 Mabsadd L
.26667 .02382 -0.91448 -0.29713 Mabsadd L
.26667 .02382 -1.46946 -2.02254 Mabsadd L
.26667 .02382 0 -0.96154 Mabsadd L
.26667 .02382 1.46946 -2.02254 Mabsadd L
.26667 .02382 .91448 -0.29713 Mabsadd L
.26667 .02382 2.37764 .77254 Mabsadd L
.26667 .02382 .56518 .7779 Mabsadd L
.26667 .02382 0 2.5 Mabsadd L closepath F
.26762 .02382 0 2.5 Mabsadd m
.26762 .02382 -0.56518 .7779 Mabsadd L
.26762 .02382 -2.37764 .77254 Mabsadd L
.26762 .02382 -0.91448 -0.29713 Mabsadd L
.26762 .02382 -1.46946 -2.02254 Mabsadd L
.26762 .02382 0 -0.96154 Mabsadd L
.26762 .02382 1.46946 -2.02254 Mabsadd L
.26762 .02382 .91448 -0.29713 Mabsadd L
.26762 .02382 2.37764 .77254 Mabsadd L
.26762 .02382 .56518 .7779 Mabsadd L
.26762 .02382 0 2.5 Mabsadd L closepath F
.26857 .02382 0 2.5 Mabsadd m
.26857 .02382 -0.56518 .7779 Mabsadd L
.26857 .02382 -2.37764 .77254 Mabsadd L
.26857 .02382 -0.91448 -0.29713 Mabsadd L
.26857 .02382 -1.46946 -2.02254 Mabsadd L
.26857 .02382 0 -0.96154 Mabsadd L
.26857 .02382 1.46946 -2.02254 Mabsadd L
.26857 .02382 .91448 -0.29713 Mabsadd L
.26857 .02382 2.37764 .77254 Mabsadd L
.26857 .02382 .56518 .7779 Mabsadd L
.26857 .02382 0 2.5 Mabsadd L closepath F
.26952 .02382 0 2.5 Mabsadd m
.26952 .02382 -0.56518 .7779 Mabsadd L
.26952 .02382 -2.37764 .77254 Mabsadd L
.26952 .02382 -0.91448 -0.29713 Mabsadd L
.26952 .02382 -1.46946 -2.02254 Mabsadd L
.26952 .02382 0 -0.96154 Mabsadd L
.26952 .02382 1.46946 -2.02254 Mabsadd L
.26952 .02382 .91448 -0.29713 Mabsadd L
.26952 .02382 2.37764 .77254 Mabsadd L
.26952 .02382 .56518 .7779 Mabsadd L
.26952 .02382 0 2.5 Mabsadd L closepath F
.27048 .02382 0 2.5 Mabsadd m
.27048 .02382 -0.56518 .7779 Mabsadd L
.27048 .02382 -2.37764 .77254 Mabsadd L
.27048 .02382 -0.91448 -0.29713 Mabsadd L
.27048 .02382 -1.46946 -2.02254 Mabsadd L
.27048 .02382 0 -0.96154 Mabsadd L
.27048 .02382 1.46946 -2.02254 Mabsadd L
.27048 .02382 .91448 -0.29713 Mabsadd L
.27048 .02382 2.37764 .77254 Mabsadd L
.27048 .02382 .56518 .7779 Mabsadd L
.27048 .02382 0 2.5 Mabsadd L closepath F
.27143 .02382 0 2.5 Mabsadd m
.27143 .02382 -0.56518 .7779 Mabsadd L
.27143 .02382 -2.37764 .77254 Mabsadd L
.27143 .02382 -0.91448 -0.29713 Mabsadd L
.27143 .02382 -1.46946 -2.02254 Mabsadd L
.27143 .02382 0 -0.96154 Mabsadd L
.27143 .02382 1.46946 -2.02254 Mabsadd L
.27143 .02382 .91448 -0.29713 Mabsadd L
.27143 .02382 2.37764 .77254 Mabsadd L
.27143 .02382 .56518 .7779 Mabsadd L
.27143 .02382 0 2.5 Mabsadd L closepath F
.27238 .02382 0 2.5 Mabsadd m
.27238 .02382 -0.56518 .7779 Mabsadd L
.27238 .02382 -2.37764 .77254 Mabsadd L
.27238 .02382 -0.91448 -0.29713 Mabsadd L
.27238 .02382 -1.46946 -2.02254 Mabsadd L
.27238 .02382 0 -0.96154 Mabsadd L
.27238 .02382 1.46946 -2.02254 Mabsadd L
.27238 .02382 .91448 -0.29713 Mabsadd L
.27238 .02382 2.37764 .77254 Mabsadd L
.27238 .02382 .56518 .7779 Mabsadd L
.27238 .02382 0 2.5 Mabsadd L closepath F
.27333 .02382 0 2.5 Mabsadd m
.27333 .02382 -0.56518 .7779 Mabsadd L
.27333 .02382 -2.37764 .77254 Mabsadd L
.27333 .02382 -0.91448 -0.29713 Mabsadd L
.27333 .02382 -1.46946 -2.02254 Mabsadd L
.27333 .02382 0 -0.96154 Mabsadd L
.27333 .02382 1.46946 -2.02254 Mabsadd L
.27333 .02382 .91448 -0.29713 Mabsadd L
.27333 .02382 2.37764 .77254 Mabsadd L
.27333 .02382 .56518 .7779 Mabsadd L
.27333 .02382 0 2.5 Mabsadd L closepath F
.27429 .02382 0 2.5 Mabsadd m
.27429 .02382 -0.56518 .7779 Mabsadd L
.27429 .02382 -2.37764 .77254 Mabsadd L
.27429 .02382 -0.91448 -0.29713 Mabsadd L
.27429 .02382 -1.46946 -2.02254 Mabsadd L
.27429 .02382 0 -0.96154 Mabsadd L
.27429 .02382 1.46946 -2.02254 Mabsadd L
.27429 .02382 .91448 -0.29713 Mabsadd L
.27429 .02382 2.37764 .77254 Mabsadd L
.27429 .02382 .56518 .7779 Mabsadd L
.27429 .02382 0 2.5 Mabsadd L closepath F
.27524 .02382 0 2.5 Mabsadd m
.27524 .02382 -0.56518 .7779 Mabsadd L
.27524 .02382 -2.37764 .77254 Mabsadd L
.27524 .02382 -0.91448 -0.29713 Mabsadd L
.27524 .02382 -1.46946 -2.02254 Mabsadd L
.27524 .02382 0 -0.96154 Mabsadd L
.27524 .02382 1.46946 -2.02254 Mabsadd L
.27524 .02382 .91448 -0.29713 Mabsadd L
.27524 .02382 2.37764 .77254 Mabsadd L
.27524 .02382 .56518 .7779 Mabsadd L
.27524 .02382 0 2.5 Mabsadd L closepath F
.27619 .02382 0 2.5 Mabsadd m
.27619 .02382 -0.56518 .7779 Mabsadd L
.27619 .02382 -2.37764 .77254 Mabsadd L
.27619 .02382 -0.91448 -0.29713 Mabsadd L
.27619 .02382 -1.46946 -2.02254 Mabsadd L
.27619 .02382 0 -0.96154 Mabsadd L
.27619 .02382 1.46946 -2.02254 Mabsadd L
.27619 .02382 .91448 -0.29713 Mabsadd L
.27619 .02382 2.37764 .77254 Mabsadd L
.27619 .02382 .56518 .7779 Mabsadd L
.27619 .02382 0 2.5 Mabsadd L closepath F
.27714 .02382 0 2.5 Mabsadd m
.27714 .02382 -0.56518 .7779 Mabsadd L
.27714 .02382 -2.37764 .77254 Mabsadd L
.27714 .02382 -0.91448 -0.29713 Mabsadd L
.27714 .02382 -1.46946 -2.02254 Mabsadd L
.27714 .02382 0 -0.96154 Mabsadd L
.27714 .02382 1.46946 -2.02254 Mabsadd L
.27714 .02382 .91448 -0.29713 Mabsadd L
.27714 .02382 2.37764 .77254 Mabsadd L
.27714 .02382 .56518 .7779 Mabsadd L
.27714 .02382 0 2.5 Mabsadd L closepath F
.2781 .02382 0 2.5 Mabsadd m
.2781 .02382 -0.56518 .7779 Mabsadd L
.2781 .02382 -2.37764 .77254 Mabsadd L
.2781 .02382 -0.91448 -0.29713 Mabsadd L
.2781 .02382 -1.46946 -2.02254 Mabsadd L
.2781 .02382 0 -0.96154 Mabsadd L
.2781 .02382 1.46946 -2.02254 Mabsadd L
.2781 .02382 .91448 -0.29713 Mabsadd L
.2781 .02382 2.37764 .77254 Mabsadd L
.2781 .02382 .56518 .7779 Mabsadd L
.2781 .02382 0 2.5 Mabsadd L closepath F
.27905 .02382 0 2.5 Mabsadd m
.27905 .02382 -0.56518 .7779 Mabsadd L
.27905 .02382 -2.37764 .77254 Mabsadd L
.27905 .02382 -0.91448 -0.29713 Mabsadd L
.27905 .02382 -1.46946 -2.02254 Mabsadd L
.27905 .02382 0 -0.96154 Mabsadd L
.27905 .02382 1.46946 -2.02254 Mabsadd L
.27905 .02382 .91448 -0.29713 Mabsadd L
.27905 .02382 2.37764 .77254 Mabsadd L
.27905 .02382 .56518 .7779 Mabsadd L
.27905 .02382 0 2.5 Mabsadd L closepath F
.28 .02382 0 2.5 Mabsadd m
.28 .02382 -0.56518 .7779 Mabsadd L
.28 .02382 -2.37764 .77254 Mabsadd L
.28 .02382 -0.91448 -0.29713 Mabsadd L
.28 .02382 -1.46946 -2.02254 Mabsadd L
.28 .02382 0 -0.96154 Mabsadd L
.28 .02382 1.46946 -2.02254 Mabsadd L
.28 .02382 .91448 -0.29713 Mabsadd L
.28 .02382 2.37764 .77254 Mabsadd L
.28 .02382 .56518 .7779 Mabsadd L
.28 .02382 0 2.5 Mabsadd L closepath F
.28095 .02382 0 2.5 Mabsadd m
.28095 .02382 -0.56518 .7779 Mabsadd L
.28095 .02382 -2.37764 .77254 Mabsadd L
.28095 .02382 -0.91448 -0.29713 Mabsadd L
.28095 .02382 -1.46946 -2.02254 Mabsadd L
.28095 .02382 0 -0.96154 Mabsadd L
.28095 .02382 1.46946 -2.02254 Mabsadd L
.28095 .02382 .91448 -0.29713 Mabsadd L
.28095 .02382 2.37764 .77254 Mabsadd L
.28095 .02382 .56518 .7779 Mabsadd L
.28095 .02382 0 2.5 Mabsadd L closepath F
.2819 .02382 0 2.5 Mabsadd m
.2819 .02382 -0.56518 .7779 Mabsadd L
.2819 .02382 -2.37764 .77254 Mabsadd L
.2819 .02382 -0.91448 -0.29713 Mabsadd L
.2819 .02382 -1.46946 -2.02254 Mabsadd L
.2819 .02382 0 -0.96154 Mabsadd L
.2819 .02382 1.46946 -2.02254 Mabsadd L
.2819 .02382 .91448 -0.29713 Mabsadd L
.2819 .02382 2.37764 .77254 Mabsadd L
.2819 .02382 .56518 .7779 Mabsadd L
.2819 .02382 0 2.5 Mabsadd L closepath F
.28286 .02382 0 2.5 Mabsadd m
.28286 .02382 -0.56518 .7779 Mabsadd L
.28286 .02382 -2.37764 .77254 Mabsadd L
.28286 .02382 -0.91448 -0.29713 Mabsadd L
.28286 .02382 -1.46946 -2.02254 Mabsadd L
.28286 .02382 0 -0.96154 Mabsadd L
.28286 .02382 1.46946 -2.02254 Mabsadd L
.28286 .02382 .91448 -0.29713 Mabsadd L
.28286 .02382 2.37764 .77254 Mabsadd L
.28286 .02382 .56518 .7779 Mabsadd L
.28286 .02382 0 2.5 Mabsadd L closepath F
.28381 .02382 0 2.5 Mabsadd m
.28381 .02382 -0.56518 .7779 Mabsadd L
.28381 .02382 -2.37764 .77254 Mabsadd L
.28381 .02382 -0.91448 -0.29713 Mabsadd L
.28381 .02382 -1.46946 -2.02254 Mabsadd L
.28381 .02382 0 -0.96154 Mabsadd L
.28381 .02382 1.46946 -2.02254 Mabsadd L
.28381 .02382 .91448 -0.29713 Mabsadd L
.28381 .02382 2.37764 .77254 Mabsadd L
.28381 .02382 .56518 .7779 Mabsadd L
.28381 .02382 0 2.5 Mabsadd L closepath F
.28476 .02382 0 2.5 Mabsadd m
.28476 .02382 -0.56518 .7779 Mabsadd L
.28476 .02382 -2.37764 .77254 Mabsadd L
.28476 .02382 -0.91448 -0.29713 Mabsadd L
.28476 .02382 -1.46946 -2.02254 Mabsadd L
.28476 .02382 0 -0.96154 Mabsadd L
.28476 .02382 1.46946 -2.02254 Mabsadd L
.28476 .02382 .91448 -0.29713 Mabsadd L
.28476 .02382 2.37764 .77254 Mabsadd L
.28476 .02382 .56518 .7779 Mabsadd L
.28476 .02382 0 2.5 Mabsadd L closepath F
.28571 .02382 0 2.5 Mabsadd m
.28571 .02382 -0.56518 .7779 Mabsadd L
.28571 .02382 -2.37764 .77254 Mabsadd L
.28571 .02382 -0.91448 -0.29713 Mabsadd L
.28571 .02382 -1.46946 -2.02254 Mabsadd L
.28571 .02382 0 -0.96154 Mabsadd L
.28571 .02382 1.46946 -2.02254 Mabsadd L
.28571 .02382 .91448 -0.29713 Mabsadd L
.28571 .02382 2.37764 .77254 Mabsadd L
.28571 .02382 .56518 .7779 Mabsadd L
.28571 .02382 0 2.5 Mabsadd L closepath F
.28667 .02382 0 2.5 Mabsadd m
.28667 .02382 -0.56518 .7779 Mabsadd L
.28667 .02382 -2.37764 .77254 Mabsadd L
.28667 .02382 -0.91448 -0.29713 Mabsadd L
.28667 .02382 -1.46946 -2.02254 Mabsadd L
.28667 .02382 0 -0.96154 Mabsadd L
.28667 .02382 1.46946 -2.02254 Mabsadd L
.28667 .02382 .91448 -0.29713 Mabsadd L
.28667 .02382 2.37764 .77254 Mabsadd L
.28667 .02382 .56518 .7779 Mabsadd L
.28667 .02382 0 2.5 Mabsadd L closepath F
.28762 .02382 0 2.5 Mabsadd m
.28762 .02382 -0.56518 .7779 Mabsadd L
.28762 .02382 -2.37764 .77254 Mabsadd L
.28762 .02382 -0.91448 -0.29713 Mabsadd L
.28762 .02382 -1.46946 -2.02254 Mabsadd L
.28762 .02382 0 -0.96154 Mabsadd L
.28762 .02382 1.46946 -2.02254 Mabsadd L
.28762 .02382 .91448 -0.29713 Mabsadd L
.28762 .02382 2.37764 .77254 Mabsadd L
.28762 .02382 .56518 .7779 Mabsadd L
.28762 .02382 0 2.5 Mabsadd L closepath F
.28857 .02382 0 2.5 Mabsadd m
.28857 .02382 -0.56518 .7779 Mabsadd L
.28857 .02382 -2.37764 .77254 Mabsadd L
.28857 .02382 -0.91448 -0.29713 Mabsadd L
.28857 .02382 -1.46946 -2.02254 Mabsadd L
.28857 .02382 0 -0.96154 Mabsadd L
.28857 .02382 1.46946 -2.02254 Mabsadd L
.28857 .02382 .91448 -0.29713 Mabsadd L
.28857 .02382 2.37764 .77254 Mabsadd L
.28857 .02382 .56518 .7779 Mabsadd L
.28857 .02382 0 2.5 Mabsadd L closepath F
.28952 .02382 0 2.5 Mabsadd m
.28952 .02382 -0.56518 .7779 Mabsadd L
.28952 .02382 -2.37764 .77254 Mabsadd L
.28952 .02382 -0.91448 -0.29713 Mabsadd L
.28952 .02382 -1.46946 -2.02254 Mabsadd L
.28952 .02382 0 -0.96154 Mabsadd L
.28952 .02382 1.46946 -2.02254 Mabsadd L
.28952 .02382 .91448 -0.29713 Mabsadd L
.28952 .02382 2.37764 .77254 Mabsadd L
.28952 .02382 .56518 .7779 Mabsadd L
.28952 .02382 0 2.5 Mabsadd L closepath F
.29048 .02382 0 2.5 Mabsadd m
.29048 .02382 -0.56518 .7779 Mabsadd L
.29048 .02382 -2.37764 .77254 Mabsadd L
.29048 .02382 -0.91448 -0.29713 Mabsadd L
.29048 .02382 -1.46946 -2.02254 Mabsadd L
.29048 .02382 0 -0.96154 Mabsadd L
.29048 .02382 1.46946 -2.02254 Mabsadd L
.29048 .02382 .91448 -0.29713 Mabsadd L
.29048 .02382 2.37764 .77254 Mabsadd L
.29048 .02382 .56518 .7779 Mabsadd L
.29048 .02382 0 2.5 Mabsadd L closepath F
.29143 .02382 0 2.5 Mabsadd m
.29143 .02382 -0.56518 .7779 Mabsadd L
.29143 .02382 -2.37764 .77254 Mabsadd L
.29143 .02382 -0.91448 -0.29713 Mabsadd L
.29143 .02382 -1.46946 -2.02254 Mabsadd L
.29143 .02382 0 -0.96154 Mabsadd L
.29143 .02382 1.46946 -2.02254 Mabsadd L
.29143 .02382 .91448 -0.29713 Mabsadd L
.29143 .02382 2.37764 .77254 Mabsadd L
.29143 .02382 .56518 .7779 Mabsadd L
.29143 .02382 0 2.5 Mabsadd L closepath F
.29238 .02382 0 2.5 Mabsadd m
.29238 .02382 -0.56518 .7779 Mabsadd L
.29238 .02382 -2.37764 .77254 Mabsadd L
.29238 .02382 -0.91448 -0.29713 Mabsadd L
.29238 .02382 -1.46946 -2.02254 Mabsadd L
.29238 .02382 0 -0.96154 Mabsadd L
.29238 .02382 1.46946 -2.02254 Mabsadd L
.29238 .02382 .91448 -0.29713 Mabsadd L
.29238 .02382 2.37764 .77254 Mabsadd L
.29238 .02382 .56518 .7779 Mabsadd L
.29238 .02382 0 2.5 Mabsadd L closepath F
.29333 .02382 0 2.5 Mabsadd m
.29333 .02382 -0.56518 .7779 Mabsadd L
.29333 .02382 -2.37764 .77254 Mabsadd L
.29333 .02382 -0.91448 -0.29713 Mabsadd L
.29333 .02382 -1.46946 -2.02254 Mabsadd L
.29333 .02382 0 -0.96154 Mabsadd L
.29333 .02382 1.46946 -2.02254 Mabsadd L
.29333 .02382 .91448 -0.29713 Mabsadd L
.29333 .02382 2.37764 .77254 Mabsadd L
.29333 .02382 .56518 .7779 Mabsadd L
.29333 .02382 0 2.5 Mabsadd L closepath F
.29429 .02382 0 2.5 Mabsadd m
.29429 .02382 -0.56518 .7779 Mabsadd L
.29429 .02382 -2.37764 .77254 Mabsadd L
.29429 .02382 -0.91448 -0.29713 Mabsadd L
.29429 .02382 -1.46946 -2.02254 Mabsadd L
.29429 .02382 0 -0.96154 Mabsadd L
.29429 .02382 1.46946 -2.02254 Mabsadd L
.29429 .02382 .91448 -0.29713 Mabsadd L
.29429 .02382 2.37764 .77254 Mabsadd L
.29429 .02382 .56518 .7779 Mabsadd L
.29429 .02382 0 2.5 Mabsadd L closepath F
.29524 .02382 0 2.5 Mabsadd m
.29524 .02382 -0.56518 .7779 Mabsadd L
.29524 .02382 -2.37764 .77254 Mabsadd L
.29524 .02382 -0.91448 -0.29713 Mabsadd L
.29524 .02382 -1.46946 -2.02254 Mabsadd L
.29524 .02382 0 -0.96154 Mabsadd L
.29524 .02382 1.46946 -2.02254 Mabsadd L
.29524 .02382 .91448 -0.29713 Mabsadd L
.29524 .02382 2.37764 .77254 Mabsadd L
.29524 .02382 .56518 .7779 Mabsadd L
.29524 .02382 0 2.5 Mabsadd L closepath F
.29619 .02382 0 2.5 Mabsadd m
.29619 .02382 -0.56518 .7779 Mabsadd L
.29619 .02382 -2.37764 .77254 Mabsadd L
.29619 .02382 -0.91448 -0.29713 Mabsadd L
.29619 .02382 -1.46946 -2.02254 Mabsadd L
.29619 .02382 0 -0.96154 Mabsadd L
.29619 .02382 1.46946 -2.02254 Mabsadd L
.29619 .02382 .91448 -0.29713 Mabsadd L
.29619 .02382 2.37764 .77254 Mabsadd L
.29619 .02382 .56518 .7779 Mabsadd L
.29619 .02382 0 2.5 Mabsadd L closepath F
.29714 .02382 0 2.5 Mabsadd m
.29714 .02382 -0.56518 .7779 Mabsadd L
.29714 .02382 -2.37764 .77254 Mabsadd L
.29714 .02382 -0.91448 -0.29713 Mabsadd L
.29714 .02382 -1.46946 -2.02254 Mabsadd L
.29714 .02382 0 -0.96154 Mabsadd L
.29714 .02382 1.46946 -2.02254 Mabsadd L
.29714 .02382 .91448 -0.29713 Mabsadd L
.29714 .02382 2.37764 .77254 Mabsadd L
.29714 .02382 .56518 .7779 Mabsadd L
.29714 .02382 0 2.5 Mabsadd L closepath F
.2981 .02382 0 2.5 Mabsadd m
.2981 .02382 -0.56518 .7779 Mabsadd L
.2981 .02382 -2.37764 .77254 Mabsadd L
.2981 .02382 -0.91448 -0.29713 Mabsadd L
.2981 .02382 -1.46946 -2.02254 Mabsadd L
.2981 .02382 0 -0.96154 Mabsadd L
.2981 .02382 1.46946 -2.02254 Mabsadd L
.2981 .02382 .91448 -0.29713 Mabsadd L
.2981 .02382 2.37764 .77254 Mabsadd L
.2981 .02382 .56518 .7779 Mabsadd L
.2981 .02382 0 2.5 Mabsadd L closepath F
.29905 .02382 0 2.5 Mabsadd m
.29905 .02382 -0.56518 .7779 Mabsadd L
.29905 .02382 -2.37764 .77254 Mabsadd L
.29905 .02382 -0.91448 -0.29713 Mabsadd L
.29905 .02382 -1.46946 -2.02254 Mabsadd L
.29905 .02382 0 -0.96154 Mabsadd L
.29905 .02382 1.46946 -2.02254 Mabsadd L
.29905 .02382 .91448 -0.29713 Mabsadd L
.29905 .02382 2.37764 .77254 Mabsadd L
.29905 .02382 .56518 .7779 Mabsadd L
.29905 .02382 0 2.5 Mabsadd L closepath F
.3 .02382 0 2.5 Mabsadd m
.3 .02382 -0.56518 .7779 Mabsadd L
.3 .02382 -2.37764 .77254 Mabsadd L
.3 .02382 -0.91448 -0.29713 Mabsadd L
.3 .02382 -1.46946 -2.02254 Mabsadd L
.3 .02382 0 -0.96154 Mabsadd L
.3 .02382 1.46946 -2.02254 Mabsadd L
.3 .02382 .91448 -0.29713 Mabsadd L
.3 .02382 2.37764 .77254 Mabsadd L
.3 .02382 .56518 .7779 Mabsadd L
.3 .02382 0 2.5 Mabsadd L closepath F
.30095 .02382 0 2.5 Mabsadd m
.30095 .02382 -0.56518 .7779 Mabsadd L
.30095 .02382 -2.37764 .77254 Mabsadd L
.30095 .02382 -0.91448 -0.29713 Mabsadd L
.30095 .02382 -1.46946 -2.02254 Mabsadd L
.30095 .02382 0 -0.96154 Mabsadd L
.30095 .02382 1.46946 -2.02254 Mabsadd L
.30095 .02382 .91448 -0.29713 Mabsadd L
.30095 .02382 2.37764 .77254 Mabsadd L
.30095 .02382 .56518 .7779 Mabsadd L
.30095 .02382 0 2.5 Mabsadd L closepath F
.3019 .02382 0 2.5 Mabsadd m
.3019 .02382 -0.56518 .7779 Mabsadd L
.3019 .02382 -2.37764 .77254 Mabsadd L
.3019 .02382 -0.91448 -0.29713 Mabsadd L
.3019 .02382 -1.46946 -2.02254 Mabsadd L
.3019 .02382 0 -0.96154 Mabsadd L
.3019 .02382 1.46946 -2.02254 Mabsadd L
.3019 .02382 .91448 -0.29713 Mabsadd L
.3019 .02382 2.37764 .77254 Mabsadd L
.3019 .02382 .56518 .7779 Mabsadd L
.3019 .02382 0 2.5 Mabsadd L closepath F
.30286 .02382 0 2.5 Mabsadd m
.30286 .02382 -0.56518 .7779 Mabsadd L
.30286 .02382 -2.37764 .77254 Mabsadd L
.30286 .02382 -0.91448 -0.29713 Mabsadd L
.30286 .02382 -1.46946 -2.02254 Mabsadd L
.30286 .02382 0 -0.96154 Mabsadd L
.30286 .02382 1.46946 -2.02254 Mabsadd L
.30286 .02382 .91448 -0.29713 Mabsadd L
.30286 .02382 2.37764 .77254 Mabsadd L
.30286 .02382 .56518 .7779 Mabsadd L
.30286 .02382 0 2.5 Mabsadd L closepath F
.30381 .02382 0 2.5 Mabsadd m
.30381 .02382 -0.56518 .7779 Mabsadd L
.30381 .02382 -2.37764 .77254 Mabsadd L
.30381 .02382 -0.91448 -0.29713 Mabsadd L
.30381 .02382 -1.46946 -2.02254 Mabsadd L
.30381 .02382 0 -0.96154 Mabsadd L
.30381 .02382 1.46946 -2.02254 Mabsadd L
.30381 .02382 .91448 -0.29713 Mabsadd L
.30381 .02382 2.37764 .77254 Mabsadd L
.30381 .02382 .56518 .7779 Mabsadd L
.30381 .02382 0 2.5 Mabsadd L closepath F
.30476 .02382 0 2.5 Mabsadd m
.30476 .02382 -0.56518 .7779 Mabsadd L
.30476 .02382 -2.37764 .77254 Mabsadd L
.30476 .02382 -0.91448 -0.29713 Mabsadd L
.30476 .02382 -1.46946 -2.02254 Mabsadd L
.30476 .02382 0 -0.96154 Mabsadd L
.30476 .02382 1.46946 -2.02254 Mabsadd L
.30476 .02382 .91448 -0.29713 Mabsadd L
.30476 .02382 2.37764 .77254 Mabsadd L
.30476 .02382 .56518 .7779 Mabsadd L
.30476 .02382 0 2.5 Mabsadd L closepath F
.30571 .02382 0 2.5 Mabsadd m
.30571 .02382 -0.56518 .7779 Mabsadd L
.30571 .02382 -2.37764 .77254 Mabsadd L
.30571 .02382 -0.91448 -0.29713 Mabsadd L
.30571 .02382 -1.46946 -2.02254 Mabsadd L
.30571 .02382 0 -0.96154 Mabsadd L
.30571 .02382 1.46946 -2.02254 Mabsadd L
.30571 .02382 .91448 -0.29713 Mabsadd L
.30571 .02382 2.37764 .77254 Mabsadd L
.30571 .02382 .56518 .7779 Mabsadd L
.30571 .02382 0 2.5 Mabsadd L closepath F
.30667 .02382 0 2.5 Mabsadd m
.30667 .02382 -0.56518 .7779 Mabsadd L
.30667 .02382 -2.37764 .77254 Mabsadd L
.30667 .02382 -0.91448 -0.29713 Mabsadd L
.30667 .02382 -1.46946 -2.02254 Mabsadd L
.30667 .02382 0 -0.96154 Mabsadd L
.30667 .02382 1.46946 -2.02254 Mabsadd L
.30667 .02382 .91448 -0.29713 Mabsadd L
.30667 .02382 2.37764 .77254 Mabsadd L
.30667 .02382 .56518 .7779 Mabsadd L
.30667 .02382 0 2.5 Mabsadd L closepath F
.30762 .02382 0 2.5 Mabsadd m
.30762 .02382 -0.56518 .7779 Mabsadd L
.30762 .02382 -2.37764 .77254 Mabsadd L
.30762 .02382 -0.91448 -0.29713 Mabsadd L
.30762 .02382 -1.46946 -2.02254 Mabsadd L
.30762 .02382 0 -0.96154 Mabsadd L
.30762 .02382 1.46946 -2.02254 Mabsadd L
.30762 .02382 .91448 -0.29713 Mabsadd L
.30762 .02382 2.37764 .77254 Mabsadd L
.30762 .02382 .56518 .7779 Mabsadd L
.30762 .02382 0 2.5 Mabsadd L closepath F
.30857 .02382 0 2.5 Mabsadd m
.30857 .02382 -0.56518 .7779 Mabsadd L
.30857 .02382 -2.37764 .77254 Mabsadd L
.30857 .02382 -0.91448 -0.29713 Mabsadd L
.30857 .02382 -1.46946 -2.02254 Mabsadd L
.30857 .02382 0 -0.96154 Mabsadd L
.30857 .02382 1.46946 -2.02254 Mabsadd L
.30857 .02382 .91448 -0.29713 Mabsadd L
.30857 .02382 2.37764 .77254 Mabsadd L
.30857 .02382 .56518 .7779 Mabsadd L
.30857 .02382 0 2.5 Mabsadd L closepath F
.30952 .02382 0 2.5 Mabsadd m
.30952 .02382 -0.56518 .7779 Mabsadd L
.30952 .02382 -2.37764 .77254 Mabsadd L
.30952 .02382 -0.91448 -0.29713 Mabsadd L
.30952 .02382 -1.46946 -2.02254 Mabsadd L
.30952 .02382 0 -0.96154 Mabsadd L
.30952 .02382 1.46946 -2.02254 Mabsadd L
.30952 .02382 .91448 -0.29713 Mabsadd L
.30952 .02382 2.37764 .77254 Mabsadd L
.30952 .02382 .56518 .7779 Mabsadd L
.30952 .02382 0 2.5 Mabsadd L closepath F
.31048 .02382 0 2.5 Mabsadd m
.31048 .02382 -0.56518 .7779 Mabsadd L
.31048 .02382 -2.37764 .77254 Mabsadd L
.31048 .02382 -0.91448 -0.29713 Mabsadd L
.31048 .02382 -1.46946 -2.02254 Mabsadd L
.31048 .02382 0 -0.96154 Mabsadd L
.31048 .02382 1.46946 -2.02254 Mabsadd L
.31048 .02382 .91448 -0.29713 Mabsadd L
.31048 .02382 2.37764 .77254 Mabsadd L
.31048 .02382 .56518 .7779 Mabsadd L
.31048 .02382 0 2.5 Mabsadd L closepath F
.31143 .02382 0 2.5 Mabsadd m
.31143 .02382 -0.56518 .7779 Mabsadd L
.31143 .02382 -2.37764 .77254 Mabsadd L
.31143 .02382 -0.91448 -0.29713 Mabsadd L
.31143 .02382 -1.46946 -2.02254 Mabsadd L
.31143 .02382 0 -0.96154 Mabsadd L
.31143 .02382 1.46946 -2.02254 Mabsadd L
.31143 .02382 .91448 -0.29713 Mabsadd L
.31143 .02382 2.37764 .77254 Mabsadd L
.31143 .02382 .56518 .7779 Mabsadd L
.31143 .02382 0 2.5 Mabsadd L closepath F
.31238 .02382 0 2.5 Mabsadd m
.31238 .02382 -0.56518 .7779 Mabsadd L
.31238 .02382 -2.37764 .77254 Mabsadd L
.31238 .02382 -0.91448 -0.29713 Mabsadd L
.31238 .02382 -1.46946 -2.02254 Mabsadd L
.31238 .02382 0 -0.96154 Mabsadd L
.31238 .02382 1.46946 -2.02254 Mabsadd L
.31238 .02382 .91448 -0.29713 Mabsadd L
.31238 .02382 2.37764 .77254 Mabsadd L
.31238 .02382 .56518 .7779 Mabsadd L
.31238 .02382 0 2.5 Mabsadd L closepath F
.31333 .02382 0 2.5 Mabsadd m
.31333 .02382 -0.56518 .7779 Mabsadd L
.31333 .02382 -2.37764 .77254 Mabsadd L
.31333 .02382 -0.91448 -0.29713 Mabsadd L
.31333 .02382 -1.46946 -2.02254 Mabsadd L
.31333 .02382 0 -0.96154 Mabsadd L
.31333 .02382 1.46946 -2.02254 Mabsadd L
.31333 .02382 .91448 -0.29713 Mabsadd L
.31333 .02382 2.37764 .77254 Mabsadd L
.31333 .02382 .56518 .7779 Mabsadd L
.31333 .02382 0 2.5 Mabsadd L closepath F
.31429 .02382 0 2.5 Mabsadd m
.31429 .02382 -0.56518 .7779 Mabsadd L
.31429 .02382 -2.37764 .77254 Mabsadd L
.31429 .02382 -0.91448 -0.29713 Mabsadd L
.31429 .02382 -1.46946 -2.02254 Mabsadd L
.31429 .02382 0 -0.96154 Mabsadd L
.31429 .02382 1.46946 -2.02254 Mabsadd L
.31429 .02382 .91448 -0.29713 Mabsadd L
.31429 .02382 2.37764 .77254 Mabsadd L
.31429 .02382 .56518 .7779 Mabsadd L
.31429 .02382 0 2.5 Mabsadd L closepath F
.31524 .02382 0 2.5 Mabsadd m
.31524 .02382 -0.56518 .7779 Mabsadd L
.31524 .02382 -2.37764 .77254 Mabsadd L
.31524 .02382 -0.91448 -0.29713 Mabsadd L
.31524 .02382 -1.46946 -2.02254 Mabsadd L
.31524 .02382 0 -0.96154 Mabsadd L
.31524 .02382 1.46946 -2.02254 Mabsadd L
.31524 .02382 .91448 -0.29713 Mabsadd L
.31524 .02382 2.37764 .77254 Mabsadd L
.31524 .02382 .56518 .7779 Mabsadd L
.31524 .02382 0 2.5 Mabsadd L closepath F
.31619 .02382 0 2.5 Mabsadd m
.31619 .02382 -0.56518 .7779 Mabsadd L
.31619 .02382 -2.37764 .77254 Mabsadd L
.31619 .02382 -0.91448 -0.29713 Mabsadd L
.31619 .02382 -1.46946 -2.02254 Mabsadd L
.31619 .02382 0 -0.96154 Mabsadd L
.31619 .02382 1.46946 -2.02254 Mabsadd L
.31619 .02382 .91448 -0.29713 Mabsadd L
.31619 .02382 2.37764 .77254 Mabsadd L
.31619 .02382 .56518 .7779 Mabsadd L
.31619 .02382 0 2.5 Mabsadd L closepath F
.31714 .02382 0 2.5 Mabsadd m
.31714 .02382 -0.56518 .7779 Mabsadd L
.31714 .02382 -2.37764 .77254 Mabsadd L
.31714 .02382 -0.91448 -0.29713 Mabsadd L
.31714 .02382 -1.46946 -2.02254 Mabsadd L
.31714 .02382 0 -0.96154 Mabsadd L
.31714 .02382 1.46946 -2.02254 Mabsadd L
.31714 .02382 .91448 -0.29713 Mabsadd L
.31714 .02382 2.37764 .77254 Mabsadd L
.31714 .02382 .56518 .7779 Mabsadd L
.31714 .02382 0 2.5 Mabsadd L closepath F
.3181 .02382 0 2.5 Mabsadd m
.3181 .02382 -0.56518 .7779 Mabsadd L
.3181 .02382 -2.37764 .77254 Mabsadd L
.3181 .02382 -0.91448 -0.29713 Mabsadd L
.3181 .02382 -1.46946 -2.02254 Mabsadd L
.3181 .02382 0 -0.96154 Mabsadd L
.3181 .02382 1.46946 -2.02254 Mabsadd L
.3181 .02382 .91448 -0.29713 Mabsadd L
.3181 .02382 2.37764 .77254 Mabsadd L
.3181 .02382 .56518 .7779 Mabsadd L
.3181 .02382 0 2.5 Mabsadd L closepath F
.31905 .02382 0 2.5 Mabsadd m
.31905 .02382 -0.56518 .7779 Mabsadd L
.31905 .02382 -2.37764 .77254 Mabsadd L
.31905 .02382 -0.91448 -0.29713 Mabsadd L
.31905 .02382 -1.46946 -2.02254 Mabsadd L
.31905 .02382 0 -0.96154 Mabsadd L
.31905 .02382 1.46946 -2.02254 Mabsadd L
.31905 .02382 .91448 -0.29713 Mabsadd L
.31905 .02382 2.37764 .77254 Mabsadd L
.31905 .02382 .56518 .7779 Mabsadd L
.31905 .02382 0 2.5 Mabsadd L closepath F
.32 .02382 0 2.5 Mabsadd m
.32 .02382 -0.56518 .7779 Mabsadd L
.32 .02382 -2.37764 .77254 Mabsadd L
.32 .02382 -0.91448 -0.29713 Mabsadd L
.32 .02382 -1.46946 -2.02254 Mabsadd L
.32 .02382 0 -0.96154 Mabsadd L
.32 .02382 1.46946 -2.02254 Mabsadd L
.32 .02382 .91448 -0.29713 Mabsadd L
.32 .02382 2.37764 .77254 Mabsadd L
.32 .02382 .56518 .7779 Mabsadd L
.32 .02382 0 2.5 Mabsadd L closepath F
.32095 .02382 0 2.5 Mabsadd m
.32095 .02382 -0.56518 .7779 Mabsadd L
.32095 .02382 -2.37764 .77254 Mabsadd L
.32095 .02382 -0.91448 -0.29713 Mabsadd L
.32095 .02382 -1.46946 -2.02254 Mabsadd L
.32095 .02382 0 -0.96154 Mabsadd L
.32095 .02382 1.46946 -2.02254 Mabsadd L
.32095 .02382 .91448 -0.29713 Mabsadd L
.32095 .02382 2.37764 .77254 Mabsadd L
.32095 .02382 .56518 .7779 Mabsadd L
.32095 .02382 0 2.5 Mabsadd L closepath F
.3219 .02382 0 2.5 Mabsadd m
.3219 .02382 -0.56518 .7779 Mabsadd L
.3219 .02382 -2.37764 .77254 Mabsadd L
.3219 .02382 -0.91448 -0.29713 Mabsadd L
.3219 .02382 -1.46946 -2.02254 Mabsadd L
.3219 .02382 0 -0.96154 Mabsadd L
.3219 .02382 1.46946 -2.02254 Mabsadd L
.3219 .02382 .91448 -0.29713 Mabsadd L
.3219 .02382 2.37764 .77254 Mabsadd L
.3219 .02382 .56518 .7779 Mabsadd L
.3219 .02382 0 2.5 Mabsadd L closepath F
.32286 .02382 0 2.5 Mabsadd m
.32286 .02382 -0.56518 .7779 Mabsadd L
.32286 .02382 -2.37764 .77254 Mabsadd L
.32286 .02382 -0.91448 -0.29713 Mabsadd L
.32286 .02382 -1.46946 -2.02254 Mabsadd L
.32286 .02382 0 -0.96154 Mabsadd L
.32286 .02382 1.46946 -2.02254 Mabsadd L
.32286 .02382 .91448 -0.29713 Mabsadd L
.32286 .02382 2.37764 .77254 Mabsadd L
.32286 .02382 .56518 .7779 Mabsadd L
.32286 .02382 0 2.5 Mabsadd L closepath F
.32381 .02382 0 2.5 Mabsadd m
.32381 .02382 -0.56518 .7779 Mabsadd L
.32381 .02382 -2.37764 .77254 Mabsadd L
.32381 .02382 -0.91448 -0.29713 Mabsadd L
.32381 .02382 -1.46946 -2.02254 Mabsadd L
.32381 .02382 0 -0.96154 Mabsadd L
.32381 .02382 1.46946 -2.02254 Mabsadd L
.32381 .02382 .91448 -0.29713 Mabsadd L
.32381 .02382 2.37764 .77254 Mabsadd L
.32381 .02382 .56518 .7779 Mabsadd L
.32381 .02382 0 2.5 Mabsadd L closepath F
.32476 .02382 0 2.5 Mabsadd m
.32476 .02382 -0.56518 .7779 Mabsadd L
.32476 .02382 -2.37764 .77254 Mabsadd L
.32476 .02382 -0.91448 -0.29713 Mabsadd L
.32476 .02382 -1.46946 -2.02254 Mabsadd L
.32476 .02382 0 -0.96154 Mabsadd L
.32476 .02382 1.46946 -2.02254 Mabsadd L
.32476 .02382 .91448 -0.29713 Mabsadd L
.32476 .02382 2.37764 .77254 Mabsadd L
.32476 .02382 .56518 .7779 Mabsadd L
.32476 .02382 0 2.5 Mabsadd L closepath F
.32571 .02382 0 2.5 Mabsadd m
.32571 .02382 -0.56518 .7779 Mabsadd L
.32571 .02382 -2.37764 .77254 Mabsadd L
.32571 .02382 -0.91448 -0.29713 Mabsadd L
.32571 .02382 -1.46946 -2.02254 Mabsadd L
.32571 .02382 0 -0.96154 Mabsadd L
.32571 .02382 1.46946 -2.02254 Mabsadd L
.32571 .02382 .91448 -0.29713 Mabsadd L
.32571 .02382 2.37764 .77254 Mabsadd L
.32571 .02382 .56518 .7779 Mabsadd L
.32571 .02382 0 2.5 Mabsadd L closepath F
.32667 .02382 0 2.5 Mabsadd m
.32667 .02382 -0.56518 .7779 Mabsadd L
.32667 .02382 -2.37764 .77254 Mabsadd L
.32667 .02382 -0.91448 -0.29713 Mabsadd L
.32667 .02382 -1.46946 -2.02254 Mabsadd L
.32667 .02382 0 -0.96154 Mabsadd L
.32667 .02382 1.46946 -2.02254 Mabsadd L
.32667 .02382 .91448 -0.29713 Mabsadd L
.32667 .02382 2.37764 .77254 Mabsadd L
.32667 .02382 .56518 .7779 Mabsadd L
.32667 .02382 0 2.5 Mabsadd L closepath F
.32762 .02382 0 2.5 Mabsadd m
.32762 .02382 -0.56518 .7779 Mabsadd L
.32762 .02382 -2.37764 .77254 Mabsadd L
.32762 .02382 -0.91448 -0.29713 Mabsadd L
.32762 .02382 -1.46946 -2.02254 Mabsadd L
.32762 .02382 0 -0.96154 Mabsadd L
.32762 .02382 1.46946 -2.02254 Mabsadd L
.32762 .02382 .91448 -0.29713 Mabsadd L
.32762 .02382 2.37764 .77254 Mabsadd L
.32762 .02382 .56518 .7779 Mabsadd L
.32762 .02382 0 2.5 Mabsadd L closepath F
.32857 .02382 0 2.5 Mabsadd m
.32857 .02382 -0.56518 .7779 Mabsadd L
.32857 .02382 -2.37764 .77254 Mabsadd L
.32857 .02382 -0.91448 -0.29713 Mabsadd L
.32857 .02382 -1.46946 -2.02254 Mabsadd L
.32857 .02382 0 -0.96154 Mabsadd L
.32857 .02382 1.46946 -2.02254 Mabsadd L
.32857 .02382 .91448 -0.29713 Mabsadd L
.32857 .02382 2.37764 .77254 Mabsadd L
.32857 .02382 .56518 .7779 Mabsadd L
.32857 .02382 0 2.5 Mabsadd L closepath F
.32952 .02382 0 2.5 Mabsadd m
.32952 .02382 -0.56518 .7779 Mabsadd L
.32952 .02382 -2.37764 .77254 Mabsadd L
.32952 .02382 -0.91448 -0.29713 Mabsadd L
.32952 .02382 -1.46946 -2.02254 Mabsadd L
.32952 .02382 0 -0.96154 Mabsadd L
.32952 .02382 1.46946 -2.02254 Mabsadd L
.32952 .02382 .91448 -0.29713 Mabsadd L
.32952 .02382 2.37764 .77254 Mabsadd L
.32952 .02382 .56518 .7779 Mabsadd L
.32952 .02382 0 2.5 Mabsadd L closepath F
.33048 .02382 0 2.5 Mabsadd m
.33048 .02382 -0.56518 .7779 Mabsadd L
.33048 .02382 -2.37764 .77254 Mabsadd L
.33048 .02382 -0.91448 -0.29713 Mabsadd L
.33048 .02382 -1.46946 -2.02254 Mabsadd L
.33048 .02382 0 -0.96154 Mabsadd L
.33048 .02382 1.46946 -2.02254 Mabsadd L
.33048 .02382 .91448 -0.29713 Mabsadd L
.33048 .02382 2.37764 .77254 Mabsadd L
.33048 .02382 .56518 .7779 Mabsadd L
.33048 .02382 0 2.5 Mabsadd L closepath F
.33143 .02382 0 2.5 Mabsadd m
.33143 .02382 -0.56518 .7779 Mabsadd L
.33143 .02382 -2.37764 .77254 Mabsadd L
.33143 .02382 -0.91448 -0.29713 Mabsadd L
.33143 .02382 -1.46946 -2.02254 Mabsadd L
.33143 .02382 0 -0.96154 Mabsadd L
.33143 .02382 1.46946 -2.02254 Mabsadd L
.33143 .02382 .91448 -0.29713 Mabsadd L
.33143 .02382 2.37764 .77254 Mabsadd L
.33143 .02382 .56518 .7779 Mabsadd L
.33143 .02382 0 2.5 Mabsadd L closepath F
.33238 .02382 0 2.5 Mabsadd m
.33238 .02382 -0.56518 .7779 Mabsadd L
.33238 .02382 -2.37764 .77254 Mabsadd L
.33238 .02382 -0.91448 -0.29713 Mabsadd L
.33238 .02382 -1.46946 -2.02254 Mabsadd L
.33238 .02382 0 -0.96154 Mabsadd L
.33238 .02382 1.46946 -2.02254 Mabsadd L
.33238 .02382 .91448 -0.29713 Mabsadd L
.33238 .02382 2.37764 .77254 Mabsadd L
.33238 .02382 .56518 .7779 Mabsadd L
.33238 .02382 0 2.5 Mabsadd L closepath F
.33333 .02382 0 2.5 Mabsadd m
.33333 .02382 -0.56518 .7779 Mabsadd L
.33333 .02382 -2.37764 .77254 Mabsadd L
.33333 .02382 -0.91448 -0.29713 Mabsadd L
.33333 .02382 -1.46946 -2.02254 Mabsadd L
.33333 .02382 0 -0.96154 Mabsadd L
.33333 .02382 1.46946 -2.02254 Mabsadd L
.33333 .02382 .91448 -0.29713 Mabsadd L
.33333 .02382 2.37764 .77254 Mabsadd L
.33333 .02382 .56518 .7779 Mabsadd L
.33333 .02382 0 2.5 Mabsadd L closepath F
.33429 .02382 0 2.5 Mabsadd m
.33429 .02382 -0.56518 .7779 Mabsadd L
.33429 .02382 -2.37764 .77254 Mabsadd L
.33429 .02382 -0.91448 -0.29713 Mabsadd L
.33429 .02382 -1.46946 -2.02254 Mabsadd L
.33429 .02382 0 -0.96154 Mabsadd L
.33429 .02382 1.46946 -2.02254 Mabsadd L
.33429 .02382 .91448 -0.29713 Mabsadd L
.33429 .02382 2.37764 .77254 Mabsadd L
.33429 .02382 .56518 .7779 Mabsadd L
.33429 .02382 0 2.5 Mabsadd L closepath F
.33524 .02382 0 2.5 Mabsadd m
.33524 .02382 -0.56518 .7779 Mabsadd L
.33524 .02382 -2.37764 .77254 Mabsadd L
.33524 .02382 -0.91448 -0.29713 Mabsadd L
.33524 .02382 -1.46946 -2.02254 Mabsadd L
.33524 .02382 0 -0.96154 Mabsadd L
.33524 .02382 1.46946 -2.02254 Mabsadd L
.33524 .02382 .91448 -0.29713 Mabsadd L
.33524 .02382 2.37764 .77254 Mabsadd L
.33524 .02382 .56518 .7779 Mabsadd L
.33524 .02382 0 2.5 Mabsadd L closepath F
.33619 .02382 0 2.5 Mabsadd m
.33619 .02382 -0.56518 .7779 Mabsadd L
.33619 .02382 -2.37764 .77254 Mabsadd L
.33619 .02382 -0.91448 -0.29713 Mabsadd L
.33619 .02382 -1.46946 -2.02254 Mabsadd L
.33619 .02382 0 -0.96154 Mabsadd L
.33619 .02382 1.46946 -2.02254 Mabsadd L
.33619 .02382 .91448 -0.29713 Mabsadd L
.33619 .02382 2.37764 .77254 Mabsadd L
.33619 .02382 .56518 .7779 Mabsadd L
.33619 .02382 0 2.5 Mabsadd L closepath F
.33714 .02382 0 2.5 Mabsadd m
.33714 .02382 -0.56518 .7779 Mabsadd L
.33714 .02382 -2.37764 .77254 Mabsadd L
.33714 .02382 -0.91448 -0.29713 Mabsadd L
.33714 .02382 -1.46946 -2.02254 Mabsadd L
.33714 .02382 0 -0.96154 Mabsadd L
.33714 .02382 1.46946 -2.02254 Mabsadd L
.33714 .02382 .91448 -0.29713 Mabsadd L
.33714 .02382 2.37764 .77254 Mabsadd L
.33714 .02382 .56518 .7779 Mabsadd L
.33714 .02382 0 2.5 Mabsadd L closepath F
.3381 .02382 0 2.5 Mabsadd m
.3381 .02382 -0.56518 .7779 Mabsadd L
.3381 .02382 -2.37764 .77254 Mabsadd L
.3381 .02382 -0.91448 -0.29713 Mabsadd L
.3381 .02382 -1.46946 -2.02254 Mabsadd L
.3381 .02382 0 -0.96154 Mabsadd L
.3381 .02382 1.46946 -2.02254 Mabsadd L
.3381 .02382 .91448 -0.29713 Mabsadd L
.3381 .02382 2.37764 .77254 Mabsadd L
.3381 .02382 .56518 .7779 Mabsadd L
.3381 .02382 0 2.5 Mabsadd L closepath F
.33905 .02382 0 2.5 Mabsadd m
.33905 .02382 -0.56518 .7779 Mabsadd L
.33905 .02382 -2.37764 .77254 Mabsadd L
.33905 .02382 -0.91448 -0.29713 Mabsadd L
.33905 .02382 -1.46946 -2.02254 Mabsadd L
.33905 .02382 0 -0.96154 Mabsadd L
.33905 .02382 1.46946 -2.02254 Mabsadd L
.33905 .02382 .91448 -0.29713 Mabsadd L
.33905 .02382 2.37764 .77254 Mabsadd L
.33905 .02382 .56518 .7779 Mabsadd L
.33905 .02382 0 2.5 Mabsadd L closepath F
.34 .02382 0 2.5 Mabsadd m
.34 .02382 -0.56518 .7779 Mabsadd L
.34 .02382 -2.37764 .77254 Mabsadd L
.34 .02382 -0.91448 -0.29713 Mabsadd L
.34 .02382 -1.46946 -2.02254 Mabsadd L
.34 .02382 0 -0.96154 Mabsadd L
.34 .02382 1.46946 -2.02254 Mabsadd L
.34 .02382 .91448 -0.29713 Mabsadd L
.34 .02382 2.37764 .77254 Mabsadd L
.34 .02382 .56518 .7779 Mabsadd L
.34 .02382 0 2.5 Mabsadd L closepath F
.34095 .02382 0 2.5 Mabsadd m
.34095 .02382 -0.56518 .7779 Mabsadd L
.34095 .02382 -2.37764 .77254 Mabsadd L
.34095 .02382 -0.91448 -0.29713 Mabsadd L
.34095 .02382 -1.46946 -2.02254 Mabsadd L
.34095 .02382 0 -0.96154 Mabsadd L
.34095 .02382 1.46946 -2.02254 Mabsadd L
.34095 .02382 .91448 -0.29713 Mabsadd L
.34095 .02382 2.37764 .77254 Mabsadd L
.34095 .02382 .56518 .7779 Mabsadd L
.34095 .02382 0 2.5 Mabsadd L closepath F
.3419 .02382 0 2.5 Mabsadd m
.3419 .02382 -0.56518 .7779 Mabsadd L
.3419 .02382 -2.37764 .77254 Mabsadd L
.3419 .02382 -0.91448 -0.29713 Mabsadd L
.3419 .02382 -1.46946 -2.02254 Mabsadd L
.3419 .02382 0 -0.96154 Mabsadd L
.3419 .02382 1.46946 -2.02254 Mabsadd L
.3419 .02382 .91448 -0.29713 Mabsadd L
.3419 .02382 2.37764 .77254 Mabsadd L
.3419 .02382 .56518 .7779 Mabsadd L
.3419 .02382 0 2.5 Mabsadd L closepath F
.34286 .02382 0 2.5 Mabsadd m
.34286 .02382 -0.56518 .7779 Mabsadd L
.34286 .02382 -2.37764 .77254 Mabsadd L
.34286 .02382 -0.91448 -0.29713 Mabsadd L
.34286 .02382 -1.46946 -2.02254 Mabsadd L
.34286 .02382 0 -0.96154 Mabsadd L
.34286 .02382 1.46946 -2.02254 Mabsadd L
.34286 .02382 .91448 -0.29713 Mabsadd L
.34286 .02382 2.37764 .77254 Mabsadd L
.34286 .02382 .56518 .7779 Mabsadd L
.34286 .02382 0 2.5 Mabsadd L closepath F
.34381 .02382 0 2.5 Mabsadd m
.34381 .02382 -0.56518 .7779 Mabsadd L
.34381 .02382 -2.37764 .77254 Mabsadd L
.34381 .02382 -0.91448 -0.29713 Mabsadd L
.34381 .02382 -1.46946 -2.02254 Mabsadd L
.34381 .02382 0 -0.96154 Mabsadd L
.34381 .02382 1.46946 -2.02254 Mabsadd L
.34381 .02382 .91448 -0.29713 Mabsadd L
.34381 .02382 2.37764 .77254 Mabsadd L
.34381 .02382 .56518 .7779 Mabsadd L
.34381 .02382 0 2.5 Mabsadd L closepath F
.34476 .02382 0 2.5 Mabsadd m
.34476 .02382 -0.56518 .7779 Mabsadd L
.34476 .02382 -2.37764 .77254 Mabsadd L
.34476 .02382 -0.91448 -0.29713 Mabsadd L
.34476 .02382 -1.46946 -2.02254 Mabsadd L
.34476 .02382 0 -0.96154 Mabsadd L
.34476 .02382 1.46946 -2.02254 Mabsadd L
.34476 .02382 .91448 -0.29713 Mabsadd L
.34476 .02382 2.37764 .77254 Mabsadd L
.34476 .02382 .56518 .7779 Mabsadd L
.34476 .02382 0 2.5 Mabsadd L closepath F
.34571 .02382 0 2.5 Mabsadd m
.34571 .02382 -0.56518 .7779 Mabsadd L
.34571 .02382 -2.37764 .77254 Mabsadd L
.34571 .02382 -0.91448 -0.29713 Mabsadd L
.34571 .02382 -1.46946 -2.02254 Mabsadd L
.34571 .02382 0 -0.96154 Mabsadd L
.34571 .02382 1.46946 -2.02254 Mabsadd L
.34571 .02382 .91448 -0.29713 Mabsadd L
.34571 .02382 2.37764 .77254 Mabsadd L
.34571 .02382 .56518 .7779 Mabsadd L
.34571 .02382 0 2.5 Mabsadd L closepath F
.34667 .02382 0 2.5 Mabsadd m
.34667 .02382 -0.56518 .7779 Mabsadd L
.34667 .02382 -2.37764 .77254 Mabsadd L
.34667 .02382 -0.91448 -0.29713 Mabsadd L
.34667 .02382 -1.46946 -2.02254 Mabsadd L
.34667 .02382 0 -0.96154 Mabsadd L
.34667 .02382 1.46946 -2.02254 Mabsadd L
.34667 .02382 .91448 -0.29713 Mabsadd L
.34667 .02382 2.37764 .77254 Mabsadd L
.34667 .02382 .56518 .7779 Mabsadd L
.34667 .02382 0 2.5 Mabsadd L closepath F
.34762 .02382 0 2.5 Mabsadd m
.34762 .02382 -0.56518 .7779 Mabsadd L
.34762 .02382 -2.37764 .77254 Mabsadd L
.34762 .02382 -0.91448 -0.29713 Mabsadd L
.34762 .02382 -1.46946 -2.02254 Mabsadd L
.34762 .02382 0 -0.96154 Mabsadd L
.34762 .02382 1.46946 -2.02254 Mabsadd L
.34762 .02382 .91448 -0.29713 Mabsadd L
.34762 .02382 2.37764 .77254 Mabsadd L
.34762 .02382 .56518 .7779 Mabsadd L
.34762 .02382 0 2.5 Mabsadd L closepath F
.34857 .02382 0 2.5 Mabsadd m
.34857 .02382 -0.56518 .7779 Mabsadd L
.34857 .02382 -2.37764 .77254 Mabsadd L
.34857 .02382 -0.91448 -0.29713 Mabsadd L
.34857 .02382 -1.46946 -2.02254 Mabsadd L
.34857 .02382 0 -0.96154 Mabsadd L
.34857 .02382 1.46946 -2.02254 Mabsadd L
.34857 .02382 .91448 -0.29713 Mabsadd L
.34857 .02382 2.37764 .77254 Mabsadd L
.34857 .02382 .56518 .7779 Mabsadd L
.34857 .02382 0 2.5 Mabsadd L closepath F
.34952 .02382 0 2.5 Mabsadd m
.34952 .02382 -0.56518 .7779 Mabsadd L
.34952 .02382 -2.37764 .77254 Mabsadd L
.34952 .02382 -0.91448 -0.29713 Mabsadd L
.34952 .02382 -1.46946 -2.02254 Mabsadd L
.34952 .02382 0 -0.96154 Mabsadd L
.34952 .02382 1.46946 -2.02254 Mabsadd L
.34952 .02382 .91448 -0.29713 Mabsadd L
.34952 .02382 2.37764 .77254 Mabsadd L
.34952 .02382 .56518 .7779 Mabsadd L
.34952 .02382 0 2.5 Mabsadd L closepath F
.35048 .02382 0 2.5 Mabsadd m
.35048 .02382 -0.56518 .7779 Mabsadd L
.35048 .02382 -2.37764 .77254 Mabsadd L
.35048 .02382 -0.91448 -0.29713 Mabsadd L
.35048 .02382 -1.46946 -2.02254 Mabsadd L
.35048 .02382 0 -0.96154 Mabsadd L
.35048 .02382 1.46946 -2.02254 Mabsadd L
.35048 .02382 .91448 -0.29713 Mabsadd L
.35048 .02382 2.37764 .77254 Mabsadd L
.35048 .02382 .56518 .7779 Mabsadd L
.35048 .02382 0 2.5 Mabsadd L closepath F
.35143 .02381 0 2.5 Mabsadd m
.35143 .02381 -0.56518 .7779 Mabsadd L
.35143 .02381 -2.37764 .77254 Mabsadd L
.35143 .02381 -0.91448 -0.29713 Mabsadd L
.35143 .02381 -1.46946 -2.02254 Mabsadd L
.35143 .02381 0 -0.96154 Mabsadd L
.35143 .02381 1.46946 -2.02254 Mabsadd L
.35143 .02381 .91448 -0.29713 Mabsadd L
.35143 .02381 2.37764 .77254 Mabsadd L
.35143 .02381 .56518 .7779 Mabsadd L
.35143 .02381 0 2.5 Mabsadd L closepath F
.35238 .02382 0 2.5 Mabsadd m
.35238 .02382 -0.56518 .7779 Mabsadd L
.35238 .02382 -2.37764 .77254 Mabsadd L
.35238 .02382 -0.91448 -0.29713 Mabsadd L
.35238 .02382 -1.46946 -2.02254 Mabsadd L
.35238 .02382 0 -0.96154 Mabsadd L
.35238 .02382 1.46946 -2.02254 Mabsadd L
.35238 .02382 .91448 -0.29713 Mabsadd L
.35238 .02382 2.37764 .77254 Mabsadd L
.35238 .02382 .56518 .7779 Mabsadd L
.35238 .02382 0 2.5 Mabsadd L closepath F
.35333 .02382 0 2.5 Mabsadd m
.35333 .02382 -0.56518 .7779 Mabsadd L
.35333 .02382 -2.37764 .77254 Mabsadd L
.35333 .02382 -0.91448 -0.29713 Mabsadd L
.35333 .02382 -1.46946 -2.02254 Mabsadd L
.35333 .02382 0 -0.96154 Mabsadd L
.35333 .02382 1.46946 -2.02254 Mabsadd L
.35333 .02382 .91448 -0.29713 Mabsadd L
.35333 .02382 2.37764 .77254 Mabsadd L
.35333 .02382 .56518 .7779 Mabsadd L
.35333 .02382 0 2.5 Mabsadd L closepath F
.35429 .02382 0 2.5 Mabsadd m
.35429 .02382 -0.56518 .7779 Mabsadd L
.35429 .02382 -2.37764 .77254 Mabsadd L
.35429 .02382 -0.91448 -0.29713 Mabsadd L
.35429 .02382 -1.46946 -2.02254 Mabsadd L
.35429 .02382 0 -0.96154 Mabsadd L
.35429 .02382 1.46946 -2.02254 Mabsadd L
.35429 .02382 .91448 -0.29713 Mabsadd L
.35429 .02382 2.37764 .77254 Mabsadd L
.35429 .02382 .56518 .7779 Mabsadd L
.35429 .02382 0 2.5 Mabsadd L closepath F
.35524 .02382 0 2.5 Mabsadd m
.35524 .02382 -0.56518 .7779 Mabsadd L
.35524 .02382 -2.37764 .77254 Mabsadd L
.35524 .02382 -0.91448 -0.29713 Mabsadd L
.35524 .02382 -1.46946 -2.02254 Mabsadd L
.35524 .02382 0 -0.96154 Mabsadd L
.35524 .02382 1.46946 -2.02254 Mabsadd L
.35524 .02382 .91448 -0.29713 Mabsadd L
.35524 .02382 2.37764 .77254 Mabsadd L
.35524 .02382 .56518 .7779 Mabsadd L
.35524 .02382 0 2.5 Mabsadd L closepath F
.35619 .02382 0 2.5 Mabsadd m
.35619 .02382 -0.56518 .7779 Mabsadd L
.35619 .02382 -2.37764 .77254 Mabsadd L
.35619 .02382 -0.91448 -0.29713 Mabsadd L
.35619 .02382 -1.46946 -2.02254 Mabsadd L
.35619 .02382 0 -0.96154 Mabsadd L
.35619 .02382 1.46946 -2.02254 Mabsadd L
.35619 .02382 .91448 -0.29713 Mabsadd L
.35619 .02382 2.37764 .77254 Mabsadd L
.35619 .02382 .56518 .7779 Mabsadd L
.35619 .02382 0 2.5 Mabsadd L closepath F
.35714 .02382 0 2.5 Mabsadd m
.35714 .02382 -0.56518 .7779 Mabsadd L
.35714 .02382 -2.37764 .77254 Mabsadd L
.35714 .02382 -0.91448 -0.29713 Mabsadd L
.35714 .02382 -1.46946 -2.02254 Mabsadd L
.35714 .02382 0 -0.96154 Mabsadd L
.35714 .02382 1.46946 -2.02254 Mabsadd L
.35714 .02382 .91448 -0.29713 Mabsadd L
.35714 .02382 2.37764 .77254 Mabsadd L
.35714 .02382 .56518 .7779 Mabsadd L
.35714 .02382 0 2.5 Mabsadd L closepath F
.3581 .02382 0 2.5 Mabsadd m
.3581 .02382 -0.56518 .7779 Mabsadd L
.3581 .02382 -2.37764 .77254 Mabsadd L
.3581 .02382 -0.91448 -0.29713 Mabsadd L
.3581 .02382 -1.46946 -2.02254 Mabsadd L
.3581 .02382 0 -0.96154 Mabsadd L
.3581 .02382 1.46946 -2.02254 Mabsadd L
.3581 .02382 .91448 -0.29713 Mabsadd L
.3581 .02382 2.37764 .77254 Mabsadd L
.3581 .02382 .56518 .7779 Mabsadd L
.3581 .02382 0 2.5 Mabsadd L closepath F
.35905 .02381 0 2.5 Mabsadd m
.35905 .02381 -0.56518 .7779 Mabsadd L
.35905 .02381 -2.37764 .77254 Mabsadd L
.35905 .02381 -0.91448 -0.29713 Mabsadd L
.35905 .02381 -1.46946 -2.02254 Mabsadd L
.35905 .02381 0 -0.96154 Mabsadd L
.35905 .02381 1.46946 -2.02254 Mabsadd L
.35905 .02381 .91448 -0.29713 Mabsadd L
.35905 .02381 2.37764 .77254 Mabsadd L
.35905 .02381 .56518 .7779 Mabsadd L
.35905 .02381 0 2.5 Mabsadd L closepath F
.36 .02382 0 2.5 Mabsadd m
.36 .02382 -0.56518 .7779 Mabsadd L
.36 .02382 -2.37764 .77254 Mabsadd L
.36 .02382 -0.91448 -0.29713 Mabsadd L
.36 .02382 -1.46946 -2.02254 Mabsadd L
.36 .02382 0 -0.96154 Mabsadd L
.36 .02382 1.46946 -2.02254 Mabsadd L
.36 .02382 .91448 -0.29713 Mabsadd L
.36 .02382 2.37764 .77254 Mabsadd L
.36 .02382 .56518 .7779 Mabsadd L
.36 .02382 0 2.5 Mabsadd L closepath F
.36095 .02381 0 2.5 Mabsadd m
.36095 .02381 -0.56518 .7779 Mabsadd L
.36095 .02381 -2.37764 .77254 Mabsadd L
.36095 .02381 -0.91448 -0.29713 Mabsadd L
.36095 .02381 -1.46946 -2.02254 Mabsadd L
.36095 .02381 0 -0.96154 Mabsadd L
.36095 .02381 1.46946 -2.02254 Mabsadd L
.36095 .02381 .91448 -0.29713 Mabsadd L
.36095 .02381 2.37764 .77254 Mabsadd L
.36095 .02381 .56518 .7779 Mabsadd L
.36095 .02381 0 2.5 Mabsadd L closepath F
.3619 .02382 0 2.5 Mabsadd m
.3619 .02382 -0.56518 .7779 Mabsadd L
.3619 .02382 -2.37764 .77254 Mabsadd L
.3619 .02382 -0.91448 -0.29713 Mabsadd L
.3619 .02382 -1.46946 -2.02254 Mabsadd L
.3619 .02382 0 -0.96154 Mabsadd L
.3619 .02382 1.46946 -2.02254 Mabsadd L
.3619 .02382 .91448 -0.29713 Mabsadd L
.3619 .02382 2.37764 .77254 Mabsadd L
.3619 .02382 .56518 .7779 Mabsadd L
.3619 .02382 0 2.5 Mabsadd L closepath F
.36286 .02382 0 2.5 Mabsadd m
.36286 .02382 -0.56518 .7779 Mabsadd L
.36286 .02382 -2.37764 .77254 Mabsadd L
.36286 .02382 -0.91448 -0.29713 Mabsadd L
.36286 .02382 -1.46946 -2.02254 Mabsadd L
.36286 .02382 0 -0.96154 Mabsadd L
.36286 .02382 1.46946 -2.02254 Mabsadd L
.36286 .02382 .91448 -0.29713 Mabsadd L
.36286 .02382 2.37764 .77254 Mabsadd L
.36286 .02382 .56518 .7779 Mabsadd L
.36286 .02382 0 2.5 Mabsadd L closepath F
.36381 .02382 0 2.5 Mabsadd m
.36381 .02382 -0.56518 .7779 Mabsadd L
.36381 .02382 -2.37764 .77254 Mabsadd L
.36381 .02382 -0.91448 -0.29713 Mabsadd L
.36381 .02382 -1.46946 -2.02254 Mabsadd L
.36381 .02382 0 -0.96154 Mabsadd L
.36381 .02382 1.46946 -2.02254 Mabsadd L
.36381 .02382 .91448 -0.29713 Mabsadd L
.36381 .02382 2.37764 .77254 Mabsadd L
.36381 .02382 .56518 .7779 Mabsadd L
.36381 .02382 0 2.5 Mabsadd L closepath F
.36476 .02382 0 2.5 Mabsadd m
.36476 .02382 -0.56518 .7779 Mabsadd L
.36476 .02382 -2.37764 .77254 Mabsadd L
.36476 .02382 -0.91448 -0.29713 Mabsadd L
.36476 .02382 -1.46946 -2.02254 Mabsadd L
.36476 .02382 0 -0.96154 Mabsadd L
.36476 .02382 1.46946 -2.02254 Mabsadd L
.36476 .02382 .91448 -0.29713 Mabsadd L
.36476 .02382 2.37764 .77254 Mabsadd L
.36476 .02382 .56518 .7779 Mabsadd L
.36476 .02382 0 2.5 Mabsadd L closepath F
.36571 .02382 0 2.5 Mabsadd m
.36571 .02382 -0.56518 .7779 Mabsadd L
.36571 .02382 -2.37764 .77254 Mabsadd L
.36571 .02382 -0.91448 -0.29713 Mabsadd L
.36571 .02382 -1.46946 -2.02254 Mabsadd L
.36571 .02382 0 -0.96154 Mabsadd L
.36571 .02382 1.46946 -2.02254 Mabsadd L
.36571 .02382 .91448 -0.29713 Mabsadd L
.36571 .02382 2.37764 .77254 Mabsadd L
.36571 .02382 .56518 .7779 Mabsadd L
.36571 .02382 0 2.5 Mabsadd L closepath F
.36667 .02382 0 2.5 Mabsadd m
.36667 .02382 -0.56518 .7779 Mabsadd L
.36667 .02382 -2.37764 .77254 Mabsadd L
.36667 .02382 -0.91448 -0.29713 Mabsadd L
.36667 .02382 -1.46946 -2.02254 Mabsadd L
.36667 .02382 0 -0.96154 Mabsadd L
.36667 .02382 1.46946 -2.02254 Mabsadd L
.36667 .02382 .91448 -0.29713 Mabsadd L
.36667 .02382 2.37764 .77254 Mabsadd L
.36667 .02382 .56518 .7779 Mabsadd L
.36667 .02382 0 2.5 Mabsadd L closepath F
.36762 .02382 0 2.5 Mabsadd m
.36762 .02382 -0.56518 .7779 Mabsadd L
.36762 .02382 -2.37764 .77254 Mabsadd L
.36762 .02382 -0.91448 -0.29713 Mabsadd L
.36762 .02382 -1.46946 -2.02254 Mabsadd L
.36762 .02382 0 -0.96154 Mabsadd L
.36762 .02382 1.46946 -2.02254 Mabsadd L
.36762 .02382 .91448 -0.29713 Mabsadd L
.36762 .02382 2.37764 .77254 Mabsadd L
.36762 .02382 .56518 .7779 Mabsadd L
.36762 .02382 0 2.5 Mabsadd L closepath F
.36857 .02382 0 2.5 Mabsadd m
.36857 .02382 -0.56518 .7779 Mabsadd L
.36857 .02382 -2.37764 .77254 Mabsadd L
.36857 .02382 -0.91448 -0.29713 Mabsadd L
.36857 .02382 -1.46946 -2.02254 Mabsadd L
.36857 .02382 0 -0.96154 Mabsadd L
.36857 .02382 1.46946 -2.02254 Mabsadd L
.36857 .02382 .91448 -0.29713 Mabsadd L
.36857 .02382 2.37764 .77254 Mabsadd L
.36857 .02382 .56518 .7779 Mabsadd L
.36857 .02382 0 2.5 Mabsadd L closepath F
.36952 .02382 0 2.5 Mabsadd m
.36952 .02382 -0.56518 .7779 Mabsadd L
.36952 .02382 -2.37764 .77254 Mabsadd L
.36952 .02382 -0.91448 -0.29713 Mabsadd L
.36952 .02382 -1.46946 -2.02254 Mabsadd L
.36952 .02382 0 -0.96154 Mabsadd L
.36952 .02382 1.46946 -2.02254 Mabsadd L
.36952 .02382 .91448 -0.29713 Mabsadd L
.36952 .02382 2.37764 .77254 Mabsadd L
.36952 .02382 .56518 .7779 Mabsadd L
.36952 .02382 0 2.5 Mabsadd L closepath F
.37048 .02382 0 2.5 Mabsadd m
.37048 .02382 -0.56518 .7779 Mabsadd L
.37048 .02382 -2.37764 .77254 Mabsadd L
.37048 .02382 -0.91448 -0.29713 Mabsadd L
.37048 .02382 -1.46946 -2.02254 Mabsadd L
.37048 .02382 0 -0.96154 Mabsadd L
.37048 .02382 1.46946 -2.02254 Mabsadd L
.37048 .02382 .91448 -0.29713 Mabsadd L
.37048 .02382 2.37764 .77254 Mabsadd L
.37048 .02382 .56518 .7779 Mabsadd L
.37048 .02382 0 2.5 Mabsadd L closepath F
.37143 .02381 0 2.5 Mabsadd m
.37143 .02381 -0.56518 .7779 Mabsadd L
.37143 .02381 -2.37764 .77254 Mabsadd L
.37143 .02381 -0.91448 -0.29713 Mabsadd L
.37143 .02381 -1.46946 -2.02254 Mabsadd L
.37143 .02381 0 -0.96154 Mabsadd L
.37143 .02381 1.46946 -2.02254 Mabsadd L
.37143 .02381 .91448 -0.29713 Mabsadd L
.37143 .02381 2.37764 .77254 Mabsadd L
.37143 .02381 .56518 .7779 Mabsadd L
.37143 .02381 0 2.5 Mabsadd L closepath F
.37238 .02381 0 2.5 Mabsadd m
.37238 .02381 -0.56518 .7779 Mabsadd L
.37238 .02381 -2.37764 .77254 Mabsadd L
.37238 .02381 -0.91448 -0.29713 Mabsadd L
.37238 .02381 -1.46946 -2.02254 Mabsadd L
.37238 .02381 0 -0.96154 Mabsadd L
.37238 .02381 1.46946 -2.02254 Mabsadd L
.37238 .02381 .91448 -0.29713 Mabsadd L
.37238 .02381 2.37764 .77254 Mabsadd L
.37238 .02381 .56518 .7779 Mabsadd L
.37238 .02381 0 2.5 Mabsadd L closepath F
.37333 .02381 0 2.5 Mabsadd m
.37333 .02381 -0.56518 .7779 Mabsadd L
.37333 .02381 -2.37764 .77254 Mabsadd L
.37333 .02381 -0.91448 -0.29713 Mabsadd L
.37333 .02381 -1.46946 -2.02254 Mabsadd L
.37333 .02381 0 -0.96154 Mabsadd L
.37333 .02381 1.46946 -2.02254 Mabsadd L
.37333 .02381 .91448 -0.29713 Mabsadd L
.37333 .02381 2.37764 .77254 Mabsadd L
.37333 .02381 .56518 .7779 Mabsadd L
.37333 .02381 0 2.5 Mabsadd L closepath F
.37429 .02381 0 2.5 Mabsadd m
.37429 .02381 -0.56518 .7779 Mabsadd L
.37429 .02381 -2.37764 .77254 Mabsadd L
.37429 .02381 -0.91448 -0.29713 Mabsadd L
.37429 .02381 -1.46946 -2.02254 Mabsadd L
.37429 .02381 0 -0.96154 Mabsadd L
.37429 .02381 1.46946 -2.02254 Mabsadd L
.37429 .02381 .91448 -0.29713 Mabsadd L
.37429 .02381 2.37764 .77254 Mabsadd L
.37429 .02381 .56518 .7779 Mabsadd L
.37429 .02381 0 2.5 Mabsadd L closepath F
.37524 .02381 0 2.5 Mabsadd m
.37524 .02381 -0.56518 .7779 Mabsadd L
.37524 .02381 -2.37764 .77254 Mabsadd L
.37524 .02381 -0.91448 -0.29713 Mabsadd L
.37524 .02381 -1.46946 -2.02254 Mabsadd L
.37524 .02381 0 -0.96154 Mabsadd L
.37524 .02381 1.46946 -2.02254 Mabsadd L
.37524 .02381 .91448 -0.29713 Mabsadd L
.37524 .02381 2.37764 .77254 Mabsadd L
.37524 .02381 .56518 .7779 Mabsadd L
.37524 .02381 0 2.5 Mabsadd L closepath F
.37619 .02381 0 2.5 Mabsadd m
.37619 .02381 -0.56518 .7779 Mabsadd L
.37619 .02381 -2.37764 .77254 Mabsadd L
.37619 .02381 -0.91448 -0.29713 Mabsadd L
.37619 .02381 -1.46946 -2.02254 Mabsadd L
.37619 .02381 0 -0.96154 Mabsadd L
.37619 .02381 1.46946 -2.02254 Mabsadd L
.37619 .02381 .91448 -0.29713 Mabsadd L
.37619 .02381 2.37764 .77254 Mabsadd L
.37619 .02381 .56518 .7779 Mabsadd L
.37619 .02381 0 2.5 Mabsadd L closepath F
.37714 .02382 0 2.5 Mabsadd m
.37714 .02382 -0.56518 .7779 Mabsadd L
.37714 .02382 -2.37764 .77254 Mabsadd L
.37714 .02382 -0.91448 -0.29713 Mabsadd L
.37714 .02382 -1.46946 -2.02254 Mabsadd L
.37714 .02382 0 -0.96154 Mabsadd L
.37714 .02382 1.46946 -2.02254 Mabsadd L
.37714 .02382 .91448 -0.29713 Mabsadd L
.37714 .02382 2.37764 .77254 Mabsadd L
.37714 .02382 .56518 .7779 Mabsadd L
.37714 .02382 0 2.5 Mabsadd L closepath F
.3781 .02382 0 2.5 Mabsadd m
.3781 .02382 -0.56518 .7779 Mabsadd L
.3781 .02382 -2.37764 .77254 Mabsadd L
.3781 .02382 -0.91448 -0.29713 Mabsadd L
.3781 .02382 -1.46946 -2.02254 Mabsadd L
.3781 .02382 0 -0.96154 Mabsadd L
.3781 .02382 1.46946 -2.02254 Mabsadd L
.3781 .02382 .91448 -0.29713 Mabsadd L
.3781 .02382 2.37764 .77254 Mabsadd L
.3781 .02382 .56518 .7779 Mabsadd L
.3781 .02382 0 2.5 Mabsadd L closepath F
.37905 .02382 0 2.5 Mabsadd m
.37905 .02382 -0.56518 .7779 Mabsadd L
.37905 .02382 -2.37764 .77254 Mabsadd L
.37905 .02382 -0.91448 -0.29713 Mabsadd L
.37905 .02382 -1.46946 -2.02254 Mabsadd L
.37905 .02382 0 -0.96154 Mabsadd L
.37905 .02382 1.46946 -2.02254 Mabsadd L
.37905 .02382 .91448 -0.29713 Mabsadd L
.37905 .02382 2.37764 .77254 Mabsadd L
.37905 .02382 .56518 .7779 Mabsadd L
.37905 .02382 0 2.5 Mabsadd L closepath F
.38 .02381 0 2.5 Mabsadd m
.38 .02381 -0.56518 .7779 Mabsadd L
.38 .02381 -2.37764 .77254 Mabsadd L
.38 .02381 -0.91448 -0.29713 Mabsadd L
.38 .02381 -1.46946 -2.02254 Mabsadd L
.38 .02381 0 -0.96154 Mabsadd L
.38 .02381 1.46946 -2.02254 Mabsadd L
.38 .02381 .91448 -0.29713 Mabsadd L
.38 .02381 2.37764 .77254 Mabsadd L
.38 .02381 .56518 .7779 Mabsadd L
.38 .02381 0 2.5 Mabsadd L closepath F
.38095 .02382 0 2.5 Mabsadd m
.38095 .02382 -0.56518 .7779 Mabsadd L
.38095 .02382 -2.37764 .77254 Mabsadd L
.38095 .02382 -0.91448 -0.29713 Mabsadd L
.38095 .02382 -1.46946 -2.02254 Mabsadd L
.38095 .02382 0 -0.96154 Mabsadd L
.38095 .02382 1.46946 -2.02254 Mabsadd L
.38095 .02382 .91448 -0.29713 Mabsadd L
.38095 .02382 2.37764 .77254 Mabsadd L
.38095 .02382 .56518 .7779 Mabsadd L
.38095 .02382 0 2.5 Mabsadd L closepath F
.3819 .02381 0 2.5 Mabsadd m
.3819 .02381 -0.56518 .7779 Mabsadd L
.3819 .02381 -2.37764 .77254 Mabsadd L
.3819 .02381 -0.91448 -0.29713 Mabsadd L
.3819 .02381 -1.46946 -2.02254 Mabsadd L
.3819 .02381 0 -0.96154 Mabsadd L
.3819 .02381 1.46946 -2.02254 Mabsadd L
.3819 .02381 .91448 -0.29713 Mabsadd L
.3819 .02381 2.37764 .77254 Mabsadd L
.3819 .02381 .56518 .7779 Mabsadd L
.3819 .02381 0 2.5 Mabsadd L closepath F
.38286 .02382 0 2.5 Mabsadd m
.38286 .02382 -0.56518 .7779 Mabsadd L
.38286 .02382 -2.37764 .77254 Mabsadd L
.38286 .02382 -0.91448 -0.29713 Mabsadd L
.38286 .02382 -1.46946 -2.02254 Mabsadd L
.38286 .02382 0 -0.96154 Mabsadd L
.38286 .02382 1.46946 -2.02254 Mabsadd L
.38286 .02382 .91448 -0.29713 Mabsadd L
.38286 .02382 2.37764 .77254 Mabsadd L
.38286 .02382 .56518 .7779 Mabsadd L
.38286 .02382 0 2.5 Mabsadd L closepath F
.38381 .02382 0 2.5 Mabsadd m
.38381 .02382 -0.56518 .7779 Mabsadd L
.38381 .02382 -2.37764 .77254 Mabsadd L
.38381 .02382 -0.91448 -0.29713 Mabsadd L
.38381 .02382 -1.46946 -2.02254 Mabsadd L
.38381 .02382 0 -0.96154 Mabsadd L
.38381 .02382 1.46946 -2.02254 Mabsadd L
.38381 .02382 .91448 -0.29713 Mabsadd L
.38381 .02382 2.37764 .77254 Mabsadd L
.38381 .02382 .56518 .7779 Mabsadd L
.38381 .02382 0 2.5 Mabsadd L closepath F
.38476 .02381 0 2.5 Mabsadd m
.38476 .02381 -0.56518 .7779 Mabsadd L
.38476 .02381 -2.37764 .77254 Mabsadd L
.38476 .02381 -0.91448 -0.29713 Mabsadd L
.38476 .02381 -1.46946 -2.02254 Mabsadd L
.38476 .02381 0 -0.96154 Mabsadd L
.38476 .02381 1.46946 -2.02254 Mabsadd L
.38476 .02381 .91448 -0.29713 Mabsadd L
.38476 .02381 2.37764 .77254 Mabsadd L
.38476 .02381 .56518 .7779 Mabsadd L
.38476 .02381 0 2.5 Mabsadd L closepath F
.38571 .02381 0 2.5 Mabsadd m
.38571 .02381 -0.56518 .7779 Mabsadd L
.38571 .02381 -2.37764 .77254 Mabsadd L
.38571 .02381 -0.91448 -0.29713 Mabsadd L
.38571 .02381 -1.46946 -2.02254 Mabsadd L
.38571 .02381 0 -0.96154 Mabsadd L
.38571 .02381 1.46946 -2.02254 Mabsadd L
.38571 .02381 .91448 -0.29713 Mabsadd L
.38571 .02381 2.37764 .77254 Mabsadd L
.38571 .02381 .56518 .7779 Mabsadd L
.38571 .02381 0 2.5 Mabsadd L closepath F
.38667 .02381 0 2.5 Mabsadd m
.38667 .02381 -0.56518 .7779 Mabsadd L
.38667 .02381 -2.37764 .77254 Mabsadd L
.38667 .02381 -0.91448 -0.29713 Mabsadd L
.38667 .02381 -1.46946 -2.02254 Mabsadd L
.38667 .02381 0 -0.96154 Mabsadd L
.38667 .02381 1.46946 -2.02254 Mabsadd L
.38667 .02381 .91448 -0.29713 Mabsadd L
.38667 .02381 2.37764 .77254 Mabsadd L
.38667 .02381 .56518 .7779 Mabsadd L
.38667 .02381 0 2.5 Mabsadd L closepath F
.38762 .02382 0 2.5 Mabsadd m
.38762 .02382 -0.56518 .7779 Mabsadd L
.38762 .02382 -2.37764 .77254 Mabsadd L
.38762 .02382 -0.91448 -0.29713 Mabsadd L
.38762 .02382 -1.46946 -2.02254 Mabsadd L
.38762 .02382 0 -0.96154 Mabsadd L
.38762 .02382 1.46946 -2.02254 Mabsadd L
.38762 .02382 .91448 -0.29713 Mabsadd L
.38762 .02382 2.37764 .77254 Mabsadd L
.38762 .02382 .56518 .7779 Mabsadd L
.38762 .02382 0 2.5 Mabsadd L closepath F
.38857 .02381 0 2.5 Mabsadd m
.38857 .02381 -0.56518 .7779 Mabsadd L
.38857 .02381 -2.37764 .77254 Mabsadd L
.38857 .02381 -0.91448 -0.29713 Mabsadd L
.38857 .02381 -1.46946 -2.02254 Mabsadd L
.38857 .02381 0 -0.96154 Mabsadd L
.38857 .02381 1.46946 -2.02254 Mabsadd L
.38857 .02381 .91448 -0.29713 Mabsadd L
.38857 .02381 2.37764 .77254 Mabsadd L
.38857 .02381 .56518 .7779 Mabsadd L
.38857 .02381 0 2.5 Mabsadd L closepath F
.38952 .02382 0 2.5 Mabsadd m
.38952 .02382 -0.56518 .7779 Mabsadd L
.38952 .02382 -2.37764 .77254 Mabsadd L
.38952 .02382 -0.91448 -0.29713 Mabsadd L
.38952 .02382 -1.46946 -2.02254 Mabsadd L
.38952 .02382 0 -0.96154 Mabsadd L
.38952 .02382 1.46946 -2.02254 Mabsadd L
.38952 .02382 .91448 -0.29713 Mabsadd L
.38952 .02382 2.37764 .77254 Mabsadd L
.38952 .02382 .56518 .7779 Mabsadd L
.38952 .02382 0 2.5 Mabsadd L closepath F
.39048 .02382 0 2.5 Mabsadd m
.39048 .02382 -0.56518 .7779 Mabsadd L
.39048 .02382 -2.37764 .77254 Mabsadd L
.39048 .02382 -0.91448 -0.29713 Mabsadd L
.39048 .02382 -1.46946 -2.02254 Mabsadd L
.39048 .02382 0 -0.96154 Mabsadd L
.39048 .02382 1.46946 -2.02254 Mabsadd L
.39048 .02382 .91448 -0.29713 Mabsadd L
.39048 .02382 2.37764 .77254 Mabsadd L
.39048 .02382 .56518 .7779 Mabsadd L
.39048 .02382 0 2.5 Mabsadd L closepath F
.39143 .02382 0 2.5 Mabsadd m
.39143 .02382 -0.56518 .7779 Mabsadd L
.39143 .02382 -2.37764 .77254 Mabsadd L
.39143 .02382 -0.91448 -0.29713 Mabsadd L
.39143 .02382 -1.46946 -2.02254 Mabsadd L
.39143 .02382 0 -0.96154 Mabsadd L
.39143 .02382 1.46946 -2.02254 Mabsadd L
.39143 .02382 .91448 -0.29713 Mabsadd L
.39143 .02382 2.37764 .77254 Mabsadd L
.39143 .02382 .56518 .7779 Mabsadd L
.39143 .02382 0 2.5 Mabsadd L closepath F
.39238 .02381 0 2.5 Mabsadd m
.39238 .02381 -0.56518 .7779 Mabsadd L
.39238 .02381 -2.37764 .77254 Mabsadd L
.39238 .02381 -0.91448 -0.29713 Mabsadd L
.39238 .02381 -1.46946 -2.02254 Mabsadd L
.39238 .02381 0 -0.96154 Mabsadd L
.39238 .02381 1.46946 -2.02254 Mabsadd L
.39238 .02381 .91448 -0.29713 Mabsadd L
.39238 .02381 2.37764 .77254 Mabsadd L
.39238 .02381 .56518 .7779 Mabsadd L
.39238 .02381 0 2.5 Mabsadd L closepath F
.39333 .02381 0 2.5 Mabsadd m
.39333 .02381 -0.56518 .7779 Mabsadd L
.39333 .02381 -2.37764 .77254 Mabsadd L
.39333 .02381 -0.91448 -0.29713 Mabsadd L
.39333 .02381 -1.46946 -2.02254 Mabsadd L
.39333 .02381 0 -0.96154 Mabsadd L
.39333 .02381 1.46946 -2.02254 Mabsadd L
.39333 .02381 .91448 -0.29713 Mabsadd L
.39333 .02381 2.37764 .77254 Mabsadd L
.39333 .02381 .56518 .7779 Mabsadd L
.39333 .02381 0 2.5 Mabsadd L closepath F
.39429 .02382 0 2.5 Mabsadd m
.39429 .02382 -0.56518 .7779 Mabsadd L
.39429 .02382 -2.37764 .77254 Mabsadd L
.39429 .02382 -0.91448 -0.29713 Mabsadd L
.39429 .02382 -1.46946 -2.02254 Mabsadd L
.39429 .02382 0 -0.96154 Mabsadd L
.39429 .02382 1.46946 -2.02254 Mabsadd L
.39429 .02382 .91448 -0.29713 Mabsadd L
.39429 .02382 2.37764 .77254 Mabsadd L
.39429 .02382 .56518 .7779 Mabsadd L
.39429 .02382 0 2.5 Mabsadd L closepath F
.39524 .02381 0 2.5 Mabsadd m
.39524 .02381 -0.56518 .7779 Mabsadd L
.39524 .02381 -2.37764 .77254 Mabsadd L
.39524 .02381 -0.91448 -0.29713 Mabsadd L
.39524 .02381 -1.46946 -2.02254 Mabsadd L
.39524 .02381 0 -0.96154 Mabsadd L
.39524 .02381 1.46946 -2.02254 Mabsadd L
.39524 .02381 .91448 -0.29713 Mabsadd L
.39524 .02381 2.37764 .77254 Mabsadd L
.39524 .02381 .56518 .7779 Mabsadd L
.39524 .02381 0 2.5 Mabsadd L closepath F
.39619 .02381 0 2.5 Mabsadd m
.39619 .02381 -0.56518 .7779 Mabsadd L
.39619 .02381 -2.37764 .77254 Mabsadd L
.39619 .02381 -0.91448 -0.29713 Mabsadd L
.39619 .02381 -1.46946 -2.02254 Mabsadd L
.39619 .02381 0 -0.96154 Mabsadd L
.39619 .02381 1.46946 -2.02254 Mabsadd L
.39619 .02381 .91448 -0.29713 Mabsadd L
.39619 .02381 2.37764 .77254 Mabsadd L
.39619 .02381 .56518 .7779 Mabsadd L
.39619 .02381 0 2.5 Mabsadd L closepath F
.39714 .02382 0 2.5 Mabsadd m
.39714 .02382 -0.56518 .7779 Mabsadd L
.39714 .02382 -2.37764 .77254 Mabsadd L
.39714 .02382 -0.91448 -0.29713 Mabsadd L
.39714 .02382 -1.46946 -2.02254 Mabsadd L
.39714 .02382 0 -0.96154 Mabsadd L
.39714 .02382 1.46946 -2.02254 Mabsadd L
.39714 .02382 .91448 -0.29713 Mabsadd L
.39714 .02382 2.37764 .77254 Mabsadd L
.39714 .02382 .56518 .7779 Mabsadd L
.39714 .02382 0 2.5 Mabsadd L closepath F
.3981 .02381 0 2.5 Mabsadd m
.3981 .02381 -0.56518 .7779 Mabsadd L
.3981 .02381 -2.37764 .77254 Mabsadd L
.3981 .02381 -0.91448 -0.29713 Mabsadd L
.3981 .02381 -1.46946 -2.02254 Mabsadd L
.3981 .02381 0 -0.96154 Mabsadd L
.3981 .02381 1.46946 -2.02254 Mabsadd L
.3981 .02381 .91448 -0.29713 Mabsadd L
.3981 .02381 2.37764 .77254 Mabsadd L
.3981 .02381 .56518 .7779 Mabsadd L
.3981 .02381 0 2.5 Mabsadd L closepath F
.39905 .02381 0 2.5 Mabsadd m
.39905 .02381 -0.56518 .7779 Mabsadd L
.39905 .02381 -2.37764 .77254 Mabsadd L
.39905 .02381 -0.91448 -0.29713 Mabsadd L
.39905 .02381 -1.46946 -2.02254 Mabsadd L
.39905 .02381 0 -0.96154 Mabsadd L
.39905 .02381 1.46946 -2.02254 Mabsadd L
.39905 .02381 .91448 -0.29713 Mabsadd L
.39905 .02381 2.37764 .77254 Mabsadd L
.39905 .02381 .56518 .7779 Mabsadd L
.39905 .02381 0 2.5 Mabsadd L closepath F
.4 .02381 0 2.5 Mabsadd m
.4 .02381 -0.56518 .7779 Mabsadd L
.4 .02381 -2.37764 .77254 Mabsadd L
.4 .02381 -0.91448 -0.29713 Mabsadd L
.4 .02381 -1.46946 -2.02254 Mabsadd L
.4 .02381 0 -0.96154 Mabsadd L
.4 .02381 1.46946 -2.02254 Mabsadd L
.4 .02381 .91448 -0.29713 Mabsadd L
.4 .02381 2.37764 .77254 Mabsadd L
.4 .02381 .56518 .7779 Mabsadd L
.4 .02381 0 2.5 Mabsadd L closepath F
.40095 .02381 0 2.5 Mabsadd m
.40095 .02381 -0.56518 .7779 Mabsadd L
.40095 .02381 -2.37764 .77254 Mabsadd L
.40095 .02381 -0.91448 -0.29713 Mabsadd L
.40095 .02381 -1.46946 -2.02254 Mabsadd L
.40095 .02381 0 -0.96154 Mabsadd L
.40095 .02381 1.46946 -2.02254 Mabsadd L
.40095 .02381 .91448 -0.29713 Mabsadd L
.40095 .02381 2.37764 .77254 Mabsadd L
.40095 .02381 .56518 .7779 Mabsadd L
.40095 .02381 0 2.5 Mabsadd L closepath F
.4019 .02381 0 2.5 Mabsadd m
.4019 .02381 -0.56518 .7779 Mabsadd L
.4019 .02381 -2.37764 .77254 Mabsadd L
.4019 .02381 -0.91448 -0.29713 Mabsadd L
.4019 .02381 -1.46946 -2.02254 Mabsadd L
.4019 .02381 0 -0.96154 Mabsadd L
.4019 .02381 1.46946 -2.02254 Mabsadd L
.4019 .02381 .91448 -0.29713 Mabsadd L
.4019 .02381 2.37764 .77254 Mabsadd L
.4019 .02381 .56518 .7779 Mabsadd L
.4019 .02381 0 2.5 Mabsadd L closepath F
.40286 .02381 0 2.5 Mabsadd m
.40286 .02381 -0.56518 .7779 Mabsadd L
.40286 .02381 -2.37764 .77254 Mabsadd L
.40286 .02381 -0.91448 -0.29713 Mabsadd L
.40286 .02381 -1.46946 -2.02254 Mabsadd L
.40286 .02381 0 -0.96154 Mabsadd L
.40286 .02381 1.46946 -2.02254 Mabsadd L
.40286 .02381 .91448 -0.29713 Mabsadd L
.40286 .02381 2.37764 .77254 Mabsadd L
.40286 .02381 .56518 .7779 Mabsadd L
.40286 .02381 0 2.5 Mabsadd L closepath F
.40381 .02381 0 2.5 Mabsadd m
.40381 .02381 -0.56518 .7779 Mabsadd L
.40381 .02381 -2.37764 .77254 Mabsadd L
.40381 .02381 -0.91448 -0.29713 Mabsadd L
.40381 .02381 -1.46946 -2.02254 Mabsadd L
.40381 .02381 0 -0.96154 Mabsadd L
.40381 .02381 1.46946 -2.02254 Mabsadd L
.40381 .02381 .91448 -0.29713 Mabsadd L
.40381 .02381 2.37764 .77254 Mabsadd L
.40381 .02381 .56518 .7779 Mabsadd L
.40381 .02381 0 2.5 Mabsadd L closepath F
.40476 .02381 0 2.5 Mabsadd m
.40476 .02381 -0.56518 .7779 Mabsadd L
.40476 .02381 -2.37764 .77254 Mabsadd L
.40476 .02381 -0.91448 -0.29713 Mabsadd L
.40476 .02381 -1.46946 -2.02254 Mabsadd L
.40476 .02381 0 -0.96154 Mabsadd L
.40476 .02381 1.46946 -2.02254 Mabsadd L
.40476 .02381 .91448 -0.29713 Mabsadd L
.40476 .02381 2.37764 .77254 Mabsadd L
.40476 .02381 .56518 .7779 Mabsadd L
.40476 .02381 0 2.5 Mabsadd L closepath F
.40571 .02381 0 2.5 Mabsadd m
.40571 .02381 -0.56518 .7779 Mabsadd L
.40571 .02381 -2.37764 .77254 Mabsadd L
.40571 .02381 -0.91448 -0.29713 Mabsadd L
.40571 .02381 -1.46946 -2.02254 Mabsadd L
.40571 .02381 0 -0.96154 Mabsadd L
.40571 .02381 1.46946 -2.02254 Mabsadd L
.40571 .02381 .91448 -0.29713 Mabsadd L
.40571 .02381 2.37764 .77254 Mabsadd L
.40571 .02381 .56518 .7779 Mabsadd L
.40571 .02381 0 2.5 Mabsadd L closepath F
.40667 .02381 0 2.5 Mabsadd m
.40667 .02381 -0.56518 .7779 Mabsadd L
.40667 .02381 -2.37764 .77254 Mabsadd L
.40667 .02381 -0.91448 -0.29713 Mabsadd L
.40667 .02381 -1.46946 -2.02254 Mabsadd L
.40667 .02381 0 -0.96154 Mabsadd L
.40667 .02381 1.46946 -2.02254 Mabsadd L
.40667 .02381 .91448 -0.29713 Mabsadd L
.40667 .02381 2.37764 .77254 Mabsadd L
.40667 .02381 .56518 .7779 Mabsadd L
.40667 .02381 0 2.5 Mabsadd L closepath F
.40762 .02381 0 2.5 Mabsadd m
.40762 .02381 -0.56518 .7779 Mabsadd L
.40762 .02381 -2.37764 .77254 Mabsadd L
.40762 .02381 -0.91448 -0.29713 Mabsadd L
.40762 .02381 -1.46946 -2.02254 Mabsadd L
.40762 .02381 0 -0.96154 Mabsadd L
.40762 .02381 1.46946 -2.02254 Mabsadd L
.40762 .02381 .91448 -0.29713 Mabsadd L
.40762 .02381 2.37764 .77254 Mabsadd L
.40762 .02381 .56518 .7779 Mabsadd L
.40762 .02381 0 2.5 Mabsadd L closepath F
.40857 .02381 0 2.5 Mabsadd m
.40857 .02381 -0.56518 .7779 Mabsadd L
.40857 .02381 -2.37764 .77254 Mabsadd L
.40857 .02381 -0.91448 -0.29713 Mabsadd L
.40857 .02381 -1.46946 -2.02254 Mabsadd L
.40857 .02381 0 -0.96154 Mabsadd L
.40857 .02381 1.46946 -2.02254 Mabsadd L
.40857 .02381 .91448 -0.29713 Mabsadd L
.40857 .02381 2.37764 .77254 Mabsadd L
.40857 .02381 .56518 .7779 Mabsadd L
.40857 .02381 0 2.5 Mabsadd L closepath F
.40952 .02381 0 2.5 Mabsadd m
.40952 .02381 -0.56518 .7779 Mabsadd L
.40952 .02381 -2.37764 .77254 Mabsadd L
.40952 .02381 -0.91448 -0.29713 Mabsadd L
.40952 .02381 -1.46946 -2.02254 Mabsadd L
.40952 .02381 0 -0.96154 Mabsadd L
.40952 .02381 1.46946 -2.02254 Mabsadd L
.40952 .02381 .91448 -0.29713 Mabsadd L
.40952 .02381 2.37764 .77254 Mabsadd L
.40952 .02381 .56518 .7779 Mabsadd L
.40952 .02381 0 2.5 Mabsadd L closepath F
.41048 .02381 0 2.5 Mabsadd m
.41048 .02381 -0.56518 .7779 Mabsadd L
.41048 .02381 -2.37764 .77254 Mabsadd L
.41048 .02381 -0.91448 -0.29713 Mabsadd L
.41048 .02381 -1.46946 -2.02254 Mabsadd L
.41048 .02381 0 -0.96154 Mabsadd L
.41048 .02381 1.46946 -2.02254 Mabsadd L
.41048 .02381 .91448 -0.29713 Mabsadd L
.41048 .02381 2.37764 .77254 Mabsadd L
.41048 .02381 .56518 .7779 Mabsadd L
.41048 .02381 0 2.5 Mabsadd L closepath F
.41143 .02381 0 2.5 Mabsadd m
.41143 .02381 -0.56518 .7779 Mabsadd L
.41143 .02381 -2.37764 .77254 Mabsadd L
.41143 .02381 -0.91448 -0.29713 Mabsadd L
.41143 .02381 -1.46946 -2.02254 Mabsadd L
.41143 .02381 0 -0.96154 Mabsadd L
.41143 .02381 1.46946 -2.02254 Mabsadd L
.41143 .02381 .91448 -0.29713 Mabsadd L
.41143 .02381 2.37764 .77254 Mabsadd L
.41143 .02381 .56518 .7779 Mabsadd L
.41143 .02381 0 2.5 Mabsadd L closepath F
.41238 .02381 0 2.5 Mabsadd m
.41238 .02381 -0.56518 .7779 Mabsadd L
.41238 .02381 -2.37764 .77254 Mabsadd L
.41238 .02381 -0.91448 -0.29713 Mabsadd L
.41238 .02381 -1.46946 -2.02254 Mabsadd L
.41238 .02381 0 -0.96154 Mabsadd L
.41238 .02381 1.46946 -2.02254 Mabsadd L
.41238 .02381 .91448 -0.29713 Mabsadd L
.41238 .02381 2.37764 .77254 Mabsadd L
.41238 .02381 .56518 .7779 Mabsadd L
.41238 .02381 0 2.5 Mabsadd L closepath F
.41333 .02381 0 2.5 Mabsadd m
.41333 .02381 -0.56518 .7779 Mabsadd L
.41333 .02381 -2.37764 .77254 Mabsadd L
.41333 .02381 -0.91448 -0.29713 Mabsadd L
.41333 .02381 -1.46946 -2.02254 Mabsadd L
.41333 .02381 0 -0.96154 Mabsadd L
.41333 .02381 1.46946 -2.02254 Mabsadd L
.41333 .02381 .91448 -0.29713 Mabsadd L
.41333 .02381 2.37764 .77254 Mabsadd L
.41333 .02381 .56518 .7779 Mabsadd L
.41333 .02381 0 2.5 Mabsadd L closepath F
.41429 .02381 0 2.5 Mabsadd m
.41429 .02381 -0.56518 .7779 Mabsadd L
.41429 .02381 -2.37764 .77254 Mabsadd L
.41429 .02381 -0.91448 -0.29713 Mabsadd L
.41429 .02381 -1.46946 -2.02254 Mabsadd L
.41429 .02381 0 -0.96154 Mabsadd L
.41429 .02381 1.46946 -2.02254 Mabsadd L
.41429 .02381 .91448 -0.29713 Mabsadd L
.41429 .02381 2.37764 .77254 Mabsadd L
.41429 .02381 .56518 .7779 Mabsadd L
.41429 .02381 0 2.5 Mabsadd L closepath F
.41524 .02381 0 2.5 Mabsadd m
.41524 .02381 -0.56518 .7779 Mabsadd L
.41524 .02381 -2.37764 .77254 Mabsadd L
.41524 .02381 -0.91448 -0.29713 Mabsadd L
.41524 .02381 -1.46946 -2.02254 Mabsadd L
.41524 .02381 0 -0.96154 Mabsadd L
.41524 .02381 1.46946 -2.02254 Mabsadd L
.41524 .02381 .91448 -0.29713 Mabsadd L
.41524 .02381 2.37764 .77254 Mabsadd L
.41524 .02381 .56518 .7779 Mabsadd L
.41524 .02381 0 2.5 Mabsadd L closepath F
.41619 .02381 0 2.5 Mabsadd m
.41619 .02381 -0.56518 .7779 Mabsadd L
.41619 .02381 -2.37764 .77254 Mabsadd L
.41619 .02381 -0.91448 -0.29713 Mabsadd L
.41619 .02381 -1.46946 -2.02254 Mabsadd L
.41619 .02381 0 -0.96154 Mabsadd L
.41619 .02381 1.46946 -2.02254 Mabsadd L
.41619 .02381 .91448 -0.29713 Mabsadd L
.41619 .02381 2.37764 .77254 Mabsadd L
.41619 .02381 .56518 .7779 Mabsadd L
.41619 .02381 0 2.5 Mabsadd L closepath F
.41714 .02381 0 2.5 Mabsadd m
.41714 .02381 -0.56518 .7779 Mabsadd L
.41714 .02381 -2.37764 .77254 Mabsadd L
.41714 .02381 -0.91448 -0.29713 Mabsadd L
.41714 .02381 -1.46946 -2.02254 Mabsadd L
.41714 .02381 0 -0.96154 Mabsadd L
.41714 .02381 1.46946 -2.02254 Mabsadd L
.41714 .02381 .91448 -0.29713 Mabsadd L
.41714 .02381 2.37764 .77254 Mabsadd L
.41714 .02381 .56518 .7779 Mabsadd L
.41714 .02381 0 2.5 Mabsadd L closepath F
.4181 .02381 0 2.5 Mabsadd m
.4181 .02381 -0.56518 .7779 Mabsadd L
.4181 .02381 -2.37764 .77254 Mabsadd L
.4181 .02381 -0.91448 -0.29713 Mabsadd L
.4181 .02381 -1.46946 -2.02254 Mabsadd L
.4181 .02381 0 -0.96154 Mabsadd L
.4181 .02381 1.46946 -2.02254 Mabsadd L
.4181 .02381 .91448 -0.29713 Mabsadd L
.4181 .02381 2.37764 .77254 Mabsadd L
.4181 .02381 .56518 .7779 Mabsadd L
.4181 .02381 0 2.5 Mabsadd L closepath F
.41905 .02381 0 2.5 Mabsadd m
.41905 .02381 -0.56518 .7779 Mabsadd L
.41905 .02381 -2.37764 .77254 Mabsadd L
.41905 .02381 -0.91448 -0.29713 Mabsadd L
.41905 .02381 -1.46946 -2.02254 Mabsadd L
.41905 .02381 0 -0.96154 Mabsadd L
.41905 .02381 1.46946 -2.02254 Mabsadd L
.41905 .02381 .91448 -0.29713 Mabsadd L
.41905 .02381 2.37764 .77254 Mabsadd L
.41905 .02381 .56518 .7779 Mabsadd L
.41905 .02381 0 2.5 Mabsadd L closepath F
.42 .02381 0 2.5 Mabsadd m
.42 .02381 -0.56518 .7779 Mabsadd L
.42 .02381 -2.37764 .77254 Mabsadd L
.42 .02381 -0.91448 -0.29713 Mabsadd L
.42 .02381 -1.46946 -2.02254 Mabsadd L
.42 .02381 0 -0.96154 Mabsadd L
.42 .02381 1.46946 -2.02254 Mabsadd L
.42 .02381 .91448 -0.29713 Mabsadd L
.42 .02381 2.37764 .77254 Mabsadd L
.42 .02381 .56518 .7779 Mabsadd L
.42 .02381 0 2.5 Mabsadd L closepath F
.42095 .02381 0 2.5 Mabsadd m
.42095 .02381 -0.56518 .7779 Mabsadd L
.42095 .02381 -2.37764 .77254 Mabsadd L
.42095 .02381 -0.91448 -0.29713 Mabsadd L
.42095 .02381 -1.46946 -2.02254 Mabsadd L
.42095 .02381 0 -0.96154 Mabsadd L
.42095 .02381 1.46946 -2.02254 Mabsadd L
.42095 .02381 .91448 -0.29713 Mabsadd L
.42095 .02381 2.37764 .77254 Mabsadd L
.42095 .02381 .56518 .7779 Mabsadd L
.42095 .02381 0 2.5 Mabsadd L closepath F
.4219 .02381 0 2.5 Mabsadd m
.4219 .02381 -0.56518 .7779 Mabsadd L
.4219 .02381 -2.37764 .77254 Mabsadd L
.4219 .02381 -0.91448 -0.29713 Mabsadd L
.4219 .02381 -1.46946 -2.02254 Mabsadd L
.4219 .02381 0 -0.96154 Mabsadd L
.4219 .02381 1.46946 -2.02254 Mabsadd L
.4219 .02381 .91448 -0.29713 Mabsadd L
.4219 .02381 2.37764 .77254 Mabsadd L
.4219 .02381 .56518 .7779 Mabsadd L
.4219 .02381 0 2.5 Mabsadd L closepath F
.42286 .02381 0 2.5 Mabsadd m
.42286 .02381 -0.56518 .7779 Mabsadd L
.42286 .02381 -2.37764 .77254 Mabsadd L
.42286 .02381 -0.91448 -0.29713 Mabsadd L
.42286 .02381 -1.46946 -2.02254 Mabsadd L
.42286 .02381 0 -0.96154 Mabsadd L
.42286 .02381 1.46946 -2.02254 Mabsadd L
.42286 .02381 .91448 -0.29713 Mabsadd L
.42286 .02381 2.37764 .77254 Mabsadd L
.42286 .02381 .56518 .7779 Mabsadd L
.42286 .02381 0 2.5 Mabsadd L closepath F
.42381 .02381 0 2.5 Mabsadd m
.42381 .02381 -0.56518 .7779 Mabsadd L
.42381 .02381 -2.37764 .77254 Mabsadd L
.42381 .02381 -0.91448 -0.29713 Mabsadd L
.42381 .02381 -1.46946 -2.02254 Mabsadd L
.42381 .02381 0 -0.96154 Mabsadd L
.42381 .02381 1.46946 -2.02254 Mabsadd L
.42381 .02381 .91448 -0.29713 Mabsadd L
.42381 .02381 2.37764 .77254 Mabsadd L
.42381 .02381 .56518 .7779 Mabsadd L
.42381 .02381 0 2.5 Mabsadd L closepath F
.42476 .02381 0 2.5 Mabsadd m
.42476 .02381 -0.56518 .7779 Mabsadd L
.42476 .02381 -2.37764 .77254 Mabsadd L
.42476 .02381 -0.91448 -0.29713 Mabsadd L
.42476 .02381 -1.46946 -2.02254 Mabsadd L
.42476 .02381 0 -0.96154 Mabsadd L
.42476 .02381 1.46946 -2.02254 Mabsadd L
.42476 .02381 .91448 -0.29713 Mabsadd L
.42476 .02381 2.37764 .77254 Mabsadd L
.42476 .02381 .56518 .7779 Mabsadd L
.42476 .02381 0 2.5 Mabsadd L closepath F
.42571 .02381 0 2.5 Mabsadd m
.42571 .02381 -0.56518 .7779 Mabsadd L
.42571 .02381 -2.37764 .77254 Mabsadd L
.42571 .02381 -0.91448 -0.29713 Mabsadd L
.42571 .02381 -1.46946 -2.02254 Mabsadd L
.42571 .02381 0 -0.96154 Mabsadd L
.42571 .02381 1.46946 -2.02254 Mabsadd L
.42571 .02381 .91448 -0.29713 Mabsadd L
.42571 .02381 2.37764 .77254 Mabsadd L
.42571 .02381 .56518 .7779 Mabsadd L
.42571 .02381 0 2.5 Mabsadd L closepath F
.42667 .02381 0 2.5 Mabsadd m
.42667 .02381 -0.56518 .7779 Mabsadd L
.42667 .02381 -2.37764 .77254 Mabsadd L
.42667 .02381 -0.91448 -0.29713 Mabsadd L
.42667 .02381 -1.46946 -2.02254 Mabsadd L
.42667 .02381 0 -0.96154 Mabsadd L
.42667 .02381 1.46946 -2.02254 Mabsadd L
.42667 .02381 .91448 -0.29713 Mabsadd L
.42667 .02381 2.37764 .77254 Mabsadd L
.42667 .02381 .56518 .7779 Mabsadd L
.42667 .02381 0 2.5 Mabsadd L closepath F
.42762 .02381 0 2.5 Mabsadd m
.42762 .02381 -0.56518 .7779 Mabsadd L
.42762 .02381 -2.37764 .77254 Mabsadd L
.42762 .02381 -0.91448 -0.29713 Mabsadd L
.42762 .02381 -1.46946 -2.02254 Mabsadd L
.42762 .02381 0 -0.96154 Mabsadd L
.42762 .02381 1.46946 -2.02254 Mabsadd L
.42762 .02381 .91448 -0.29713 Mabsadd L
.42762 .02381 2.37764 .77254 Mabsadd L
.42762 .02381 .56518 .7779 Mabsadd L
.42762 .02381 0 2.5 Mabsadd L closepath F
.42857 .02381 0 2.5 Mabsadd m
.42857 .02381 -0.56518 .7779 Mabsadd L
.42857 .02381 -2.37764 .77254 Mabsadd L
.42857 .02381 -0.91448 -0.29713 Mabsadd L
.42857 .02381 -1.46946 -2.02254 Mabsadd L
.42857 .02381 0 -0.96154 Mabsadd L
.42857 .02381 1.46946 -2.02254 Mabsadd L
.42857 .02381 .91448 -0.29713 Mabsadd L
.42857 .02381 2.37764 .77254 Mabsadd L
.42857 .02381 .56518 .7779 Mabsadd L
.42857 .02381 0 2.5 Mabsadd L closepath F
.42952 .02381 0 2.5 Mabsadd m
.42952 .02381 -0.56518 .7779 Mabsadd L
.42952 .02381 -2.37764 .77254 Mabsadd L
.42952 .02381 -0.91448 -0.29713 Mabsadd L
.42952 .02381 -1.46946 -2.02254 Mabsadd L
.42952 .02381 0 -0.96154 Mabsadd L
.42952 .02381 1.46946 -2.02254 Mabsadd L
.42952 .02381 .91448 -0.29713 Mabsadd L
.42952 .02381 2.37764 .77254 Mabsadd L
.42952 .02381 .56518 .7779 Mabsadd L
.42952 .02381 0 2.5 Mabsadd L closepath F
.43048 .02381 0 2.5 Mabsadd m
.43048 .02381 -0.56518 .7779 Mabsadd L
.43048 .02381 -2.37764 .77254 Mabsadd L
.43048 .02381 -0.91448 -0.29713 Mabsadd L
.43048 .02381 -1.46946 -2.02254 Mabsadd L
.43048 .02381 0 -0.96154 Mabsadd L
.43048 .02381 1.46946 -2.02254 Mabsadd L
.43048 .02381 .91448 -0.29713 Mabsadd L
.43048 .02381 2.37764 .77254 Mabsadd L
.43048 .02381 .56518 .7779 Mabsadd L
.43048 .02381 0 2.5 Mabsadd L closepath F
.43143 .02381 0 2.5 Mabsadd m
.43143 .02381 -0.56518 .7779 Mabsadd L
.43143 .02381 -2.37764 .77254 Mabsadd L
.43143 .02381 -0.91448 -0.29713 Mabsadd L
.43143 .02381 -1.46946 -2.02254 Mabsadd L
.43143 .02381 0 -0.96154 Mabsadd L
.43143 .02381 1.46946 -2.02254 Mabsadd L
.43143 .02381 .91448 -0.29713 Mabsadd L
.43143 .02381 2.37764 .77254 Mabsadd L
.43143 .02381 .56518 .7779 Mabsadd L
.43143 .02381 0 2.5 Mabsadd L closepath F
.43238 .02381 0 2.5 Mabsadd m
.43238 .02381 -0.56518 .7779 Mabsadd L
.43238 .02381 -2.37764 .77254 Mabsadd L
.43238 .02381 -0.91448 -0.29713 Mabsadd L
.43238 .02381 -1.46946 -2.02254 Mabsadd L
.43238 .02381 0 -0.96154 Mabsadd L
.43238 .02381 1.46946 -2.02254 Mabsadd L
.43238 .02381 .91448 -0.29713 Mabsadd L
.43238 .02381 2.37764 .77254 Mabsadd L
.43238 .02381 .56518 .7779 Mabsadd L
.43238 .02381 0 2.5 Mabsadd L closepath F
.43333 .02381 0 2.5 Mabsadd m
.43333 .02381 -0.56518 .7779 Mabsadd L
.43333 .02381 -2.37764 .77254 Mabsadd L
.43333 .02381 -0.91448 -0.29713 Mabsadd L
.43333 .02381 -1.46946 -2.02254 Mabsadd L
.43333 .02381 0 -0.96154 Mabsadd L
.43333 .02381 1.46946 -2.02254 Mabsadd L
.43333 .02381 .91448 -0.29713 Mabsadd L
.43333 .02381 2.37764 .77254 Mabsadd L
.43333 .02381 .56518 .7779 Mabsadd L
.43333 .02381 0 2.5 Mabsadd L closepath F
.43429 .02381 0 2.5 Mabsadd m
.43429 .02381 -0.56518 .7779 Mabsadd L
.43429 .02381 -2.37764 .77254 Mabsadd L
.43429 .02381 -0.91448 -0.29713 Mabsadd L
.43429 .02381 -1.46946 -2.02254 Mabsadd L
.43429 .02381 0 -0.96154 Mabsadd L
.43429 .02381 1.46946 -2.02254 Mabsadd L
.43429 .02381 .91448 -0.29713 Mabsadd L
.43429 .02381 2.37764 .77254 Mabsadd L
.43429 .02381 .56518 .7779 Mabsadd L
.43429 .02381 0 2.5 Mabsadd L closepath F
.43524 .02381 0 2.5 Mabsadd m
.43524 .02381 -0.56518 .7779 Mabsadd L
.43524 .02381 -2.37764 .77254 Mabsadd L
.43524 .02381 -0.91448 -0.29713 Mabsadd L
.43524 .02381 -1.46946 -2.02254 Mabsadd L
.43524 .02381 0 -0.96154 Mabsadd L
.43524 .02381 1.46946 -2.02254 Mabsadd L
.43524 .02381 .91448 -0.29713 Mabsadd L
.43524 .02381 2.37764 .77254 Mabsadd L
.43524 .02381 .56518 .7779 Mabsadd L
.43524 .02381 0 2.5 Mabsadd L closepath F
.43619 .02381 0 2.5 Mabsadd m
.43619 .02381 -0.56518 .7779 Mabsadd L
.43619 .02381 -2.37764 .77254 Mabsadd L
.43619 .02381 -0.91448 -0.29713 Mabsadd L
.43619 .02381 -1.46946 -2.02254 Mabsadd L
.43619 .02381 0 -0.96154 Mabsadd L
.43619 .02381 1.46946 -2.02254 Mabsadd L
.43619 .02381 .91448 -0.29713 Mabsadd L
.43619 .02381 2.37764 .77254 Mabsadd L
.43619 .02381 .56518 .7779 Mabsadd L
.43619 .02381 0 2.5 Mabsadd L closepath F
.43714 .02381 0 2.5 Mabsadd m
.43714 .02381 -0.56518 .7779 Mabsadd L
.43714 .02381 -2.37764 .77254 Mabsadd L
.43714 .02381 -0.91448 -0.29713 Mabsadd L
.43714 .02381 -1.46946 -2.02254 Mabsadd L
.43714 .02381 0 -0.96154 Mabsadd L
.43714 .02381 1.46946 -2.02254 Mabsadd L
.43714 .02381 .91448 -0.29713 Mabsadd L
.43714 .02381 2.37764 .77254 Mabsadd L
.43714 .02381 .56518 .7779 Mabsadd L
.43714 .02381 0 2.5 Mabsadd L closepath F
.4381 .02381 0 2.5 Mabsadd m
.4381 .02381 -0.56518 .7779 Mabsadd L
.4381 .02381 -2.37764 .77254 Mabsadd L
.4381 .02381 -0.91448 -0.29713 Mabsadd L
.4381 .02381 -1.46946 -2.02254 Mabsadd L
.4381 .02381 0 -0.96154 Mabsadd L
.4381 .02381 1.46946 -2.02254 Mabsadd L
.4381 .02381 .91448 -0.29713 Mabsadd L
.4381 .02381 2.37764 .77254 Mabsadd L
.4381 .02381 .56518 .7779 Mabsadd L
.4381 .02381 0 2.5 Mabsadd L closepath F
.43905 .02381 0 2.5 Mabsadd m
.43905 .02381 -0.56518 .7779 Mabsadd L
.43905 .02381 -2.37764 .77254 Mabsadd L
.43905 .02381 -0.91448 -0.29713 Mabsadd L
.43905 .02381 -1.46946 -2.02254 Mabsadd L
.43905 .02381 0 -0.96154 Mabsadd L
.43905 .02381 1.46946 -2.02254 Mabsadd L
.43905 .02381 .91448 -0.29713 Mabsadd L
.43905 .02381 2.37764 .77254 Mabsadd L
.43905 .02381 .56518 .7779 Mabsadd L
.43905 .02381 0 2.5 Mabsadd L closepath F
.44 .02381 0 2.5 Mabsadd m
.44 .02381 -0.56518 .7779 Mabsadd L
.44 .02381 -2.37764 .77254 Mabsadd L
.44 .02381 -0.91448 -0.29713 Mabsadd L
.44 .02381 -1.46946 -2.02254 Mabsadd L
.44 .02381 0 -0.96154 Mabsadd L
.44 .02381 1.46946 -2.02254 Mabsadd L
.44 .02381 .91448 -0.29713 Mabsadd L
.44 .02381 2.37764 .77254 Mabsadd L
.44 .02381 .56518 .7779 Mabsadd L
.44 .02381 0 2.5 Mabsadd L closepath F
.44095 .02381 0 2.5 Mabsadd m
.44095 .02381 -0.56518 .7779 Mabsadd L
.44095 .02381 -2.37764 .77254 Mabsadd L
.44095 .02381 -0.91448 -0.29713 Mabsadd L
.44095 .02381 -1.46946 -2.02254 Mabsadd L
.44095 .02381 0 -0.96154 Mabsadd L
.44095 .02381 1.46946 -2.02254 Mabsadd L
.44095 .02381 .91448 -0.29713 Mabsadd L
.44095 .02381 2.37764 .77254 Mabsadd L
.44095 .02381 .56518 .7779 Mabsadd L
.44095 .02381 0 2.5 Mabsadd L closepath F
.4419 .02381 0 2.5 Mabsadd m
.4419 .02381 -0.56518 .7779 Mabsadd L
.4419 .02381 -2.37764 .77254 Mabsadd L
.4419 .02381 -0.91448 -0.29713 Mabsadd L
.4419 .02381 -1.46946 -2.02254 Mabsadd L
.4419 .02381 0 -0.96154 Mabsadd L
.4419 .02381 1.46946 -2.02254 Mabsadd L
.4419 .02381 .91448 -0.29713 Mabsadd L
.4419 .02381 2.37764 .77254 Mabsadd L
.4419 .02381 .56518 .7779 Mabsadd L
.4419 .02381 0 2.5 Mabsadd L closepath F
.44286 .02381 0 2.5 Mabsadd m
.44286 .02381 -0.56518 .7779 Mabsadd L
.44286 .02381 -2.37764 .77254 Mabsadd L
.44286 .02381 -0.91448 -0.29713 Mabsadd L
.44286 .02381 -1.46946 -2.02254 Mabsadd L
.44286 .02381 0 -0.96154 Mabsadd L
.44286 .02381 1.46946 -2.02254 Mabsadd L
.44286 .02381 .91448 -0.29713 Mabsadd L
.44286 .02381 2.37764 .77254 Mabsadd L
.44286 .02381 .56518 .7779 Mabsadd L
.44286 .02381 0 2.5 Mabsadd L closepath F
.44381 .02381 0 2.5 Mabsadd m
.44381 .02381 -0.56518 .7779 Mabsadd L
.44381 .02381 -2.37764 .77254 Mabsadd L
.44381 .02381 -0.91448 -0.29713 Mabsadd L
.44381 .02381 -1.46946 -2.02254 Mabsadd L
.44381 .02381 0 -0.96154 Mabsadd L
.44381 .02381 1.46946 -2.02254 Mabsadd L
.44381 .02381 .91448 -0.29713 Mabsadd L
.44381 .02381 2.37764 .77254 Mabsadd L
.44381 .02381 .56518 .7779 Mabsadd L
.44381 .02381 0 2.5 Mabsadd L closepath F
.44476 .02381 0 2.5 Mabsadd m
.44476 .02381 -0.56518 .7779 Mabsadd L
.44476 .02381 -2.37764 .77254 Mabsadd L
.44476 .02381 -0.91448 -0.29713 Mabsadd L
.44476 .02381 -1.46946 -2.02254 Mabsadd L
.44476 .02381 0 -0.96154 Mabsadd L
.44476 .02381 1.46946 -2.02254 Mabsadd L
.44476 .02381 .91448 -0.29713 Mabsadd L
.44476 .02381 2.37764 .77254 Mabsadd L
.44476 .02381 .56518 .7779 Mabsadd L
.44476 .02381 0 2.5 Mabsadd L closepath F
.44571 .02381 0 2.5 Mabsadd m
.44571 .02381 -0.56518 .7779 Mabsadd L
.44571 .02381 -2.37764 .77254 Mabsadd L
.44571 .02381 -0.91448 -0.29713 Mabsadd L
.44571 .02381 -1.46946 -2.02254 Mabsadd L
.44571 .02381 0 -0.96154 Mabsadd L
.44571 .02381 1.46946 -2.02254 Mabsadd L
.44571 .02381 .91448 -0.29713 Mabsadd L
.44571 .02381 2.37764 .77254 Mabsadd L
.44571 .02381 .56518 .7779 Mabsadd L
.44571 .02381 0 2.5 Mabsadd L closepath F
.44667 .02381 0 2.5 Mabsadd m
.44667 .02381 -0.56518 .7779 Mabsadd L
.44667 .02381 -2.37764 .77254 Mabsadd L
.44667 .02381 -0.91448 -0.29713 Mabsadd L
.44667 .02381 -1.46946 -2.02254 Mabsadd L
.44667 .02381 0 -0.96154 Mabsadd L
.44667 .02381 1.46946 -2.02254 Mabsadd L
.44667 .02381 .91448 -0.29713 Mabsadd L
.44667 .02381 2.37764 .77254 Mabsadd L
.44667 .02381 .56518 .7779 Mabsadd L
.44667 .02381 0 2.5 Mabsadd L closepath F
.44762 .02381 0 2.5 Mabsadd m
.44762 .02381 -0.56518 .7779 Mabsadd L
.44762 .02381 -2.37764 .77254 Mabsadd L
.44762 .02381 -0.91448 -0.29713 Mabsadd L
.44762 .02381 -1.46946 -2.02254 Mabsadd L
.44762 .02381 0 -0.96154 Mabsadd L
.44762 .02381 1.46946 -2.02254 Mabsadd L
.44762 .02381 .91448 -0.29713 Mabsadd L
.44762 .02381 2.37764 .77254 Mabsadd L
.44762 .02381 .56518 .7779 Mabsadd L
.44762 .02381 0 2.5 Mabsadd L closepath F
.44857 .02381 0 2.5 Mabsadd m
.44857 .02381 -0.56518 .7779 Mabsadd L
.44857 .02381 -2.37764 .77254 Mabsadd L
.44857 .02381 -0.91448 -0.29713 Mabsadd L
.44857 .02381 -1.46946 -2.02254 Mabsadd L
.44857 .02381 0 -0.96154 Mabsadd L
.44857 .02381 1.46946 -2.02254 Mabsadd L
.44857 .02381 .91448 -0.29713 Mabsadd L
.44857 .02381 2.37764 .77254 Mabsadd L
.44857 .02381 .56518 .7779 Mabsadd L
.44857 .02381 0 2.5 Mabsadd L closepath F
.44952 .02381 0 2.5 Mabsadd m
.44952 .02381 -0.56518 .7779 Mabsadd L
.44952 .02381 -2.37764 .77254 Mabsadd L
.44952 .02381 -0.91448 -0.29713 Mabsadd L
.44952 .02381 -1.46946 -2.02254 Mabsadd L
.44952 .02381 0 -0.96154 Mabsadd L
.44952 .02381 1.46946 -2.02254 Mabsadd L
.44952 .02381 .91448 -0.29713 Mabsadd L
.44952 .02381 2.37764 .77254 Mabsadd L
.44952 .02381 .56518 .7779 Mabsadd L
.44952 .02381 0 2.5 Mabsadd L closepath F
.45048 .02381 0 2.5 Mabsadd m
.45048 .02381 -0.56518 .7779 Mabsadd L
.45048 .02381 -2.37764 .77254 Mabsadd L
.45048 .02381 -0.91448 -0.29713 Mabsadd L
.45048 .02381 -1.46946 -2.02254 Mabsadd L
.45048 .02381 0 -0.96154 Mabsadd L
.45048 .02381 1.46946 -2.02254 Mabsadd L
.45048 .02381 .91448 -0.29713 Mabsadd L
.45048 .02381 2.37764 .77254 Mabsadd L
.45048 .02381 .56518 .7779 Mabsadd L
.45048 .02381 0 2.5 Mabsadd L closepath F
.45143 .02381 0 2.5 Mabsadd m
.45143 .02381 -0.56518 .7779 Mabsadd L
.45143 .02381 -2.37764 .77254 Mabsadd L
.45143 .02381 -0.91448 -0.29713 Mabsadd L
.45143 .02381 -1.46946 -2.02254 Mabsadd L
.45143 .02381 0 -0.96154 Mabsadd L
.45143 .02381 1.46946 -2.02254 Mabsadd L
.45143 .02381 .91448 -0.29713 Mabsadd L
.45143 .02381 2.37764 .77254 Mabsadd L
.45143 .02381 .56518 .7779 Mabsadd L
.45143 .02381 0 2.5 Mabsadd L closepath F
.45238 .02381 0 2.5 Mabsadd m
.45238 .02381 -0.56518 .7779 Mabsadd L
.45238 .02381 -2.37764 .77254 Mabsadd L
.45238 .02381 -0.91448 -0.29713 Mabsadd L
.45238 .02381 -1.46946 -2.02254 Mabsadd L
.45238 .02381 0 -0.96154 Mabsadd L
.45238 .02381 1.46946 -2.02254 Mabsadd L
.45238 .02381 .91448 -0.29713 Mabsadd L
.45238 .02381 2.37764 .77254 Mabsadd L
.45238 .02381 .56518 .7779 Mabsadd L
.