(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.0'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     91545,       2575]*)
(*NotebookOutlinePosition[     92776,       2622]*)
(*  CellTagsIndexPosition[     92634,       2613]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["LocallyExact", "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"s:1"],

Cell[TextData[{
  "The solution of a differential system can sometimes be solvable by \
analytic means. The function ",
  StyleBox["DSolve", "MR"],
  " implements many of the known algorithmic techniques."
}], "Text"],

Cell[TextData[{
  "However, differential systems that can be solved in closed form constitute \
only a small subset. Despite this fact, when a closed-form solution does not \
exist for the entire vector field, it is often possible to analytically solve \
a system of differential equations for part of the vector field. An example \
of this is the method ",
  StyleBox["Splitting", "MR"],
  ", which breaks up a vector field ",
  Cell[BoxData[
      \(TraditionalForm\`f\)]],
  " into subfields",
  Cell[BoxData[
      \(TraditionalForm\`\(\(\ \)\(f\_1\ , \[Ellipsis], \ f\_n\)\)\)]],
  " such that ",
  Cell[BoxData[
      \(TraditionalForm\`f\  = \ f\_1\  + \ \[CenterEllipsis]\  + \ f\_n\)]],
  "."
}], "Text"],

Cell[TextData[{
  "The idea underlying the method ",
  StyleBox["LocallyExact", "MR"],
  " is that rather than using a standard numerical integration scheme, when a \
solution can be found by ",
  StyleBox["DSolve", "MR"],
  ", direct numerical evaluation can be used to locally advance the \
solution."
}], "Text"],

Cell[TextData[{
  "Since the method ",
  StyleBox["LocallyExact", "MR"],
  " makes no attempt to adaptively adjust step sizes, it is primarily \
intended for use as a submethod between integration steps."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
  CellTags->"s:2"],

Cell["Load a package with some predefined problems.", "Text"],

Cell[BoxData[
    \(\(Needs["\<DifferentialEquations`NDSolveProblems`\>"];\)\)], "Input",
  CellLabel->"In[1]:="],

Cell[CellGroupData[{

Cell["Harmonic oscillator", "Subsection"],

Cell[TextData[{
  "Numerically solve the equations of motion for a harmonic oscillator using \
the method ",
  StyleBox["LocallyExact", "MR"],
  ". The result is two interpolating functions that approximate the solution \
and the first derivative."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(system\  = \ 
        GetNDSolveProblem["\<HarmonicOscillator\>"];\)\), \
"\[IndentingNewLine]", 
    \(\(vars\  = \ 
        system["\<DependentVariables\>"[]];\)\), "\[IndentingNewLine]", 
    \(\(\(tdata\  = \ system["\<TimeData\>"[]];\)\(\[IndentingNewLine]\)
    \)\), "\[IndentingNewLine]", 
    \(sols\  = \ 
      vars\  /. \ First[
          NDSolve[system, \ StartingStepSize \[Rule] 1/10, \ 
            Method \[Rule] "\<LocallyExact\>"]]\)}], "Input",
  CellLabel->"In[2]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          TagBox[\(InterpolatingFunction[{{0.`, 10.`}}, "<>"]\),
            False,
            Editable->False], "[", "T", "]"}], ",", 
        RowBox[{
          TagBox[\(InterpolatingFunction[{{0.`, 10.`}}, "<>"]\),
            False,
            Editable->False], "[", "T", "]"}]}], "}"}]], "Output",
  CellLabel->"Out[5]="]
}, Open  ]],

Cell["The solution evolves on the unit circle.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ParametricPlot[Evaluate[sols], Evaluate[tdata], 
      AspectRatio \[Rule] 1]\)], "Input",
  CellLabel->"In[6]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.476191 0.5 0.476191 [
[.02381 .4875 -6 -9 ]
[.02381 .4875 6 0 ]
[.2619 .4875 -12 -9 ]
[.2619 .4875 12 0 ]
[.7381 .4875 -9 -9 ]
[.7381 .4875 9 0 ]
[.97619 .4875 -3 -9 ]
[.97619 .4875 3 0 ]
[.4875 .02381 -12 -4.5 ]
[.4875 .02381 0 4.5 ]
[.4875 .2619 -24 -4.5 ]
[.4875 .2619 0 4.5 ]
[.4875 .7381 -18 -4.5 ]
[.4875 .7381 0 4.5 ]
[.4875 .97619 -6 -4.5 ]
[.4875 .97619 0 4.5 ]
[ 0 0 0 0 ]
[ 1 1 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 .5 m
.02381 .50625 L
s
[(-1)] .02381 .4875 0 1 Mshowa
.2619 .5 m
.2619 .50625 L
s
[(-0.5)] .2619 .4875 0 1 Mshowa
.7381 .5 m
.7381 .50625 L
s
[(0.5)] .7381 .4875 0 1 Mshowa
.97619 .5 m
.97619 .50625 L
s
[(1)] .97619 .4875 0 1 Mshowa
.125 Mabswid
.07143 .5 m
.07143 .50375 L
s
.11905 .5 m
.11905 .50375 L
s
.16667 .5 m
.16667 .50375 L
s
.21429 .5 m
.21429 .50375 L
s
.30952 .5 m
.30952 .50375 L
s
.35714 .5 m
.35714 .50375 L
s
.40476 .5 m
.40476 .50375 L
s
.45238 .5 m
.45238 .50375 L
s
.54762 .5 m
.54762 .50375 L
s
.59524 .5 m
.59524 .50375 L
s
.64286 .5 m
.64286 .50375 L
s
.69048 .5 m
.69048 .50375 L
s
.78571 .5 m
.78571 .50375 L
s
.83333 .5 m
.83333 .50375 L
s
.88095 .5 m
.88095 .50375 L
s
.92857 .5 m
.92857 .50375 L
s
.25 Mabswid
0 .5 m
1 .5 L
s
.5 .02381 m
.50625 .02381 L
s
[(-1)] .4875 .02381 1 0 Mshowa
.5 .2619 m
.50625 .2619 L
s
[(-0.5)] .4875 .2619 1 0 Mshowa
.5 .7381 m
.50625 .7381 L
s
[(0.5)] .4875 .7381 1 0 Mshowa
.5 .97619 m
.50625 .97619 L
s
[(1)] .4875 .97619 1 0 Mshowa
.125 Mabswid
.5 .07143 m
.50375 .07143 L
s
.5 .11905 m
.50375 .11905 L
s
.5 .16667 m
.50375 .16667 L
s
.5 .21429 m
.50375 .21429 L
s
.5 .30952 m
.50375 .30952 L
s
.5 .35714 m
.50375 .35714 L
s
.5 .40476 m
.50375 .40476 L
s
.5 .45238 m
.50375 .45238 L
s
.5 .54762 m
.50375 .54762 L
s
.5 .59524 m
.50375 .59524 L
s
.5 .64286 m
.50375 .64286 L
s
.5 .69048 m
.50375 .69048 L
s
.5 .78571 m
.50375 .78571 L
s
.5 .83333 m
.50375 .83333 L
s
.5 .88095 m
.50375 .88095 L
s
.5 .92857 m
.50375 .92857 L
s
.25 Mabswid
.5 0 m
.5 1 L
s
0 0 m
1 0 L
1 1 L
0 1 L
closepath
clip
newpath
.5 Mabswid
.97619 .5 m
.97615 .49416 L
.97605 .48832 L
.97562 .47664 L
.9739 .45334 L
.96705 .40713 L
.9369 .31059 L
.89091 .22807 L
.83088 .15755 L
.7517 .09577 L
.66762 .05428 L
.56927 .02887 L
.54447 .02589 L
.53203 .02489 L
.51956 .02421 L
.51331 .024 L
.50707 .02386 L
.50083 .02381 L
.49458 .02384 L
.48834 .02395 L
.4821 .02415 L
.46962 .02478 L
.45801 .02566 L
.44642 .02683 L
.42334 .03002 L
.37779 .03976 L
.28359 .07583 L
.20433 .12672 L
.13784 .19081 L
.08118 .27341 L
.04498 .3596 L
.03262 .4088 L
.02843 .43382 L
.02557 .45903 L
.02467 .47145 L
.02408 .4839 L
.02391 .49012 L
.02382 .49635 L
.02382 .50258 L
.02389 .5088 L
.02405 .51503 L
.02428 .52125 L
.025 .53369 L
.02741 .55848 L
.03536 .60424 L
.04772 .64901 L
.08912 .7407 L
.14436 .81666 L
.21974 .88499 L
.30623 .93498 L
Mistroke
.39468 .9644 L
.44455 .97295 L
.46975 .97523 L
.48238 .97587 L
.48871 .97606 L
.49503 .97617 L
.50094 .97619 L
.50685 .97614 L
.51275 .97602 L
.51866 .97583 L
.54224 .97431 L
.58902 .9678 L
.67779 .94176 L
.7664 .8947 L
.8385 .83492 L
.90178 .7556 L
.94617 .6664 L
.97 .57656 L
.97339 .55156 L
.97459 .53899 L
.97546 .5264 L
.97599 .5138 L
.97613 .50749 L
.97619 .50118 L
.97616 .49487 L
.97605 .48856 L
.97558 .47595 L
.97484 .46419 L
.97381 .45246 L
.97088 .42908 L
.96157 .38291 L
.93034 .29612 L
.87823 .21069 L
.81449 .14244 L
.73183 .08405 L
.6451 .04646 L
.60028 .03449 L
.57748 .03016 L
.5545 .02694 L
.52943 .02472 L
.51686 .02411 L
.51057 .02393 L
.50428 .02383 L
.49799 .02381 L
.4917 .02388 L
.47912 .02427 L
.45401 .02603 L
.40754 .03287 L
.36198 .04425 L
.2684 .08392 L
Mistroke
.18677 .14133 L
.12296 .20914 L
.07012 .29515 L
.0382 .38383 L
.0291 .42919 L
.02622 .45214 L
.02446 .47521 L
.02397 .48775 L
.02385 .49403 L
.02381 .5003 L
.02385 .50658 L
.02398 .51285 L
.02449 .5254 L
.02649 .55042 L
.0298 .5753 L
.0536 .66577 L
.0988 .75651 L
.10044 .75906 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[6]:=",
  ImageSize->{288, 288},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P000181000`40n007h00OSooo00<00?ooool0
S_oo002?ool00`00ooooo`2>ool008ooo`03003ooooo08koo`00Sooo00<00?ooool0S_oo002?ool0
0`00ooooo`2>ool008Koo`@000Goo`03003ooooo08koo`00R?oo00<00?ooool01?oo00<00?ooool0
S_oo0026oolA0029ool007goo``000Koo`03003ooooo00Coo``007koo`00Mooo20002Ooo00<00?oo
ool01?oo00<00?ooool03_oo2@00Mooo001bool6000>ool30006ool00`00ooooo`0Cool9001bool0
06koo`D001Goo`03003ooooo00Coo`03003ooooo01Ooo`@00004ool00000001_ool006[oo`D0023o
o`03003ooooo01_oo`P006_oo`00I_oo1@009?oo00<00?ooool07ooo1`00J?oo001Tool4000Wool0
0`00ooooo`0Sool6001Uool006;oo`@002Woo`03003ooooo02Ooo`@006?oo`00H?oo0`00;?oo00<0
0?ooool0:ooo0`00H?oo001Nool2000_ool00`00ooooo`0]ool3001Nool005_oo`<0037oo`03003o
oooo02ooo`<005coo`00F?oo1000<ooo00<00?ooool0<_oo0P00F_oo001Fool3000fool00`00oooo
o`0dool2001Hool005?oo`@003Soo`8003Ooo`<005Goo`00DOoo1000>_oo00<00?ooool0>?oo0`00
Dooo001?ool4000lool00`00ooooo`0jool3001Aool004koo`<003koo`03003ooooo03coo`<004oo
o`00C?oo0`00@?oo00<00?ooool0?_oo0`00COoo001;ool20012ool00`00ooooo`10ool3001;ool0
04Woo`8004Coo`03003ooooo04;oo`8004[oo`00B?oo0P00AOoo00<00?ooool0A?oo0P00B?oo0016
ool20017ool00`00ooooo`16ool00`00ooooo`15ool004Coo`<004Soo`03003ooooo04Ooo`8004Go
o`00@ooo0P00B_oo00<00?ooool0B?oo0P00A?oo0011ool3001;ool00`00ooooo`1:ool00`00oooo
o`11ool0043oo`8004goo`03003ooooo04_oo`80047oo`00?_oo0`00C_oo00<00?ooool0COoo00<0
0?ooool0?_oo000mool2001@ool2001?ool2000nool003_oo`<0057oo`03003ooooo04ooo`8003go
o`00>_oo0P00Dooo00<00?ooool0D?oo0P00??oo000iool2001Dool00`00ooooo`1Bool2000jool0
03Soo`03003ooooo05Coo`03003ooooo05?oo`8003Woo`00=_oo0P00Eooo00<00?ooool0E?oo0`00
=ooo000eool2001Hool00`00ooooo`1Eool3000fool003Coo`8005Woo`03003ooooo05Ooo`8003Go
o`00<ooo0P00F_oo00<00?ooool0F?oo0P00=?oo000bool2001Kool00`00ooooo`1Iool2000cool0
037oo`8005coo`03003ooooo05_oo`03003ooooo033oo`00<?oo0P00GOoo00<00?ooool0G?oo00<0
0?ooool0;ooo000_ool2001Nool00`00ooooo`1Mool00`00ooooo`0^ool002koo`8005ooo`8005oo
o`03003ooooo02goo`00;Ooo0P00H?oo00<00?ooool0Gooo00<00?ooool0;?oo000/ool2001Qool0
0`00ooooo`1Oool2000]ool002_oo`8006;oo`03003ooooo063oo`03003ooooo02_oo`00:_oo0P00
Hooo00<00?ooool0HOoo00<00?ooool0:_oo000Yool2001Tool00`00ooooo`1Rool00`00ooooo`0Y
ool002Soo`8006Goo`03003ooooo06?oo`03003ooooo02Soo`009ooo0P00I_oo00<00?ooool0I?oo
00<00?ooool09ooo000Vool00`00ool0001Vool00`00ooooo`1Tool2000Xool002Koo`8006Ooo`03
003ooooo06Goo`8002Ooo`009Ooo0P00J?oo00<00?ooool0I_oo0P009_oo000Tool2001Yool00`00
ooooo`1Wool2000Uool002Coo`03003ooooo06Soo`03003ooooo06Soo`8002Coo`008ooo00<00?oo
ool0JOoo00<00?ooool0JOoo0P008ooo000Rool00`00ooooo`1Zool2001[ool2000Rool002;oo`03
003ooooo06[oo`03003ooooo06[oo`8002;oo`008Ooo00<00?ooool0Jooo00<00?ooool0Jooo0P00
8Ooo000Pool2001]ool00`00ooooo`1/ool00`00ooooo`0Oool0023oo`03003ooooo06coo`03003o
oooo06goo`03003ooooo01koo`007ooo00<00?ooool0KOoo00<00?ooool0K_oo00<00?ooool07Ooo
000Nool2001_ool00`00ooooo`1^ool00`00ooooo`0Mool001koo`03003ooooo06koo`03003ooooo
06ooo`03003ooooo01coo`007Ooo00<00?ooool0Kooo00<00?ooool0Kooo0P007Ooo000Lool2001a
ool00`00ooooo`1`ool00`00ooooo`0Kool001coo`03003ooooo073oo`03003ooooo073oo`8001co
o`006ooo0P00L_oo00<00?ooool0LOoo00<00?ooool06_oo000Jool2001Oool20004ool00`00oooo
o`02ool30006ool00`00ooooo`1aool2000Kool001[oo`03003ooooo05goo`04003ooooo000;ool0
0`00ooooo`03ool00`00ooooo`1bool2000Jool001Woo`8005ooo`04003ooooo000;ool00`00oooo
o`03ool3001bool2000Jool001Soo`8005[oo`@000;oo`04003ooooo0008ool30006ool00`00oooo
o`1cool2000Iool001Soo`03003ooooo05ooo`04003ooooo0008ool00`00ooooo`06ool00`00oooo
o`1cool00`00ool0000Hool001Ooo`80067oo`04003ooooo0008ool00`00ooooo`06ool00`00oooo
o`1dool2000Hool001Ooo`03003ooooo067oo`8000Woo`@000Goo`03003ooooo07Goo`8001Ooo`00
5_oo0P00Mooo00<00?ooool0MOoo0P005ooo000Fool00`00ooooo`1fool00`00ooooo`1fool2000F
ool001Goo`03003ooooo07Ooo`03003ooooo07Koo`03003oo`0001Goo`005Ooo00<00?ooool0Mooo
00<00?ooool0Mooo0P005Ooo000Dool00`00ooooo`1hool00`00ooooo`1gool00`00ool0000Dool0
01Coo`03003ooooo07Soo`03003ooooo07Soo`8001Coo`004ooo00<00?ooool0NOoo00<00?ooool0
N?oo00<00?oo00004ooo000Cool00`00ooooo`1iool00`00ooooo`1iool2000Cool001;oo`8007_o
o`8007[oo`8001?oo`004_oo00<00?ooool0N_oo00<00?ooool0N_oo0P004_oo000Bool00`00oooo
o`1jool00`00ooooo`1jool2000Bool0017oo`03003ooooo07_oo`03003ooooo07_oo`03003ooooo
013oo`004Ooo00<00?ooool0Nooo00<00?ooool0Nooo0P004Ooo000@ool2001mool00`00ooooo`1k
ool2000Aool0013oo`03003ooooo07coo`03003ooooo07coo`03003ooooo00ooo`004?oo00<00?oo
ool0O?oo00<00?ooool0O?oo0P004?oo000?ool00`00ooooo`1mool00`00ooooo`1lool2000@ool0
00ooo`03003ooooo07goo`03003ooooo07goo`03003ooooo00koo`003_oo0P00Oooo00<00?ooool0
OOoo00<00?ooool03_oo000>ool00`00ooooo`1nool00`00ooooo`1mool2000?ool000goo`80083o
o`03003ooooo07koo`03003ooooo00goo`003Ooo00<00?ooool0Oooo00<00?ooool0O_oo00<00?oo
ool03Ooo000<ool20021ool2001oool2000>ool000coo`80087oo`03003ooooo07ooo`03003ooooo
00coo`003?oo00<00?ooool0P?oo00<00?ooool0Oooo00<00?ooool03?oo000<ool00`00ooooo`20
ool00`00ooooo`1oool2000=ool000_oo`8008;oo`03003ooooo083oo`03003ooooo00_oo`002ooo
00<00?ooool0POoo00<00?ooool0P?oo00<00?ooool02ooo000;ool00`00ooooo`21ool00`00oooo
o`20ool2000<ool000[oo`8008?oo`03003ooooo087oo`03003ooooo00[oo`002_oo0P00Pooo00<0
0?ooool0POoo00<00?ooool02_oo000:ool00`00ooooo`22ool00`00ooooo`21ool00`00ooooo`0:
ool000[oo`03003ooooo08;oo`03003ooooo08;oo`03003ooooo00Woo`002_oo00<00?ooool0P_oo
00<00?ooool0P_oo00<00?ooool02Ooo0009ool20024ool00`00ooooo`22ool00`00ooooo`09ool0
00Woo`03003ooooo08?oo`03003ooooo08;oo`8000[oo`002Ooo00<00?ooool0Pooo0P00Pooo0P00
2_oo0009ool00`00ooooo`23ool00`00ooooo`23ool00`00ooooo`08ool000Woo`03003ooooo08?o
o`03003ooooo08?oo`03003ooooo00Soo`002Ooo00<00?ooool0Pooo00<00?ooool0Pooo00<00?oo
ool02?oo0008ool20025ool00`00ooooo`23ool20009ool000Soo`03003ooooo08Coo`03003ooooo
08Coo`03003ooooo00Ooo`002?oo00<00?ooool0Q?oo00<00?ooool0Q?oo00<00?ooool01ooo0008
ool00`00ooooo`24ool00`00ooooo`24ool00`00ooooo`07ool000Soo`03003ooooo08Coo`03003o
oooo08Coo`03003ooooo00Ooo`002?oo00<00?ooool0Q?oo00<00?ooool0Q?oo00<00?ooool01ooo
0008ool00`00ooooo`24ool00`00ooooo`24ool00`00ooooo`07ool000Ooo`8008Koo`03003ooooo
08Goo`03003ooooo00Koo`001ooo00<00?ooool0QOoo00<00?ooool0QOoo00<00?ooool01_oo0007
ool00`00ooooo`25ool20026ool00`00ooooo`06ool000Ooo`03003ooooo08Goo`03003ooooo08Go
o`03003ooooo00Koo`001ooo00<00?ooool0QOoo00<00?ooool0QOoo00<00?ooool01_oo0007ool4
000lool20004ool00`00ooooo`02ool3000jool00`00ooooo`0jool20004ool00`00ooooo`02ool3
000lool40006ool000Ooo`03003oo`0003coo`04003ooooo000;ool00`00ooooo`0gool00`00oooo
o`0iool01000ooooo`002ooo00<00?ooool0>_oo0P001ooo0007ool00`00ool0000lool01000oooo
o`002ooo00<00?ooool0=ooo00<00?ooool0>Ooo00@00?ooool000_oo`03003ooooo03[oo`8000Oo
o`0000?oo`0000000P000_oo00<00?oo0000=_oo10000_oo00@00?ooool000Soo`<003[oo`03003o
oooo03Woo`04003ooooo0008ool3000mool20007ool000Ooo`03003oo`0003coo`04003ooooo0008
ool00`00ooooo`0jool00`00ooooo`0iool01000ooooo`002?oo00<00?ooool0?Ooo0P001ooo0006
ool4000lool01000ooooo`002?oo00<00?ooool0>_oo00<00?ooool0>Ooo00@00?ooool000Soo`03
003ooooo03coo`<000Ooo`001_oo00@00?ooool003goo`8000Woo`@003Woo`03003ooooo03[oo`80
00Woo`@003coo`8000Ooo`001_oo00<00?ooool0Q_oo00<00?ooool0Q_oo00<00?ooool01Ooo0006
ool00`00ooooo`26ool00`00ooooo`26ool00`00ooooo`05ool000Koo`03003ooooo08Koo`03003o
oooo08Koo`03003ooooo00Goo`001_oo00<00?ooool0Q_oo00<00?ooool0Q_oo00<00?ooool01Ooo
003o000Q000000Koo`03003ooooo00_oo`03003ooooo00_oo`03003ooooo00[oo`03003ooooo00_o
o`03003ooooo00_oo`03003ooooo00[oo`03003ooooo00_oo`03003ooooo00_oo`03003ooooo00[o
o`03003ooooo00_oo`03003ooooo00_oo`03003ooooo00[oo`03003ooooo00_oo`03003ooooo00_o
o`03003ooooo00[oo`03003ooooo00_oo`03003ooooo00_oo`03003ooooo00[oo`03003ooooo00_o
o`03003ooooo00_oo`03003ooooo00Goo`001_oo00<00?ooool0@_oo00<00?ooool0@Ooo00<00?oo
ool0@Ooo00<00?ooool0@_oo00<00?ooool01Ooo0006ool00`00ooooo`26ool00`00ooooo`26ool0
0`00ooooo`05ool000Koo`03003ooooo08Koo`03003ooooo08Koo`03003ooooo00Goo`001_oo00<0
0?ooool0Q_oo00<00?ooool0Q_oo00<00?ooool01Ooo0006ool20027ool00`00ooooo`26ool00`00
ooooo`05ool000Koo`8008Ooo`03003ooooo08Goo`03003ooooo00Koo`001_oo0P00Qooo00<00?oo
ool0QOoo00<00?ooool01_oo0007ool00`00ooooo`25ool00`00ooooo`25ool00`00ooooo`06ool0
00Ooo`03003ooooo08Goo`03003ooooo08Goo`03003ooooo00Koo`001ooo00<00?ooool0QOoo00<0
0?ooool0QOoo00<00?ooool01_oo0007ool00`00ooooo`25ool00`00ooooo`25ool00`00ooooo`06
ool000Ooo`03003ooooo08Goo`03003ooooo08Goo`03003ooooo00Koo`001ooo00<00?ooool0QOoo
0P00Q_oo00<00?ooool01_oo0007ool00`00ooooo`25ool00`00ooooo`25ool00`00ooooo`06ool0
00Ooo`03003ooooo08Goo`03003ooooo08Goo`03003ooooo00Koo`001ooo00<00?ooool0QOoo00<0
0?ooool0QOoo00<00?ooool01_oo0007ool00`00ooooo`25ool00`00ooooo`25ool00`00ooooo`06
ool000Ooo`8008Koo`03003ooooo08Coo`03003ooooo00Ooo`002?oo00<00?ooool0Q?oo00<00?oo
ool0Q?oo00<00?ooool01ooo0008ool00`00ooooo`24ool00`00ooooo`24ool00`00ooooo`07ool0
00Soo`03003ooooo08Coo`03003ooooo08Coo`03003ooooo00Ooo`002?oo00<00?ooool0Q?oo00<0
0?ooool0Q?oo00<00?ooool01ooo0009ool00`00ooooo`23ool00`00ooooo`23ool00`00ooooo`08
ool000Woo`03003ooooo08?oo`03003ooooo08?oo`03003ooooo00Soo`002Ooo00<00?ooool0Pooo
00<00?ooool0Pooo00<00?ooool02?oo0009ool00`00ooooo`23ool20024ool00`00ooooo`08ool0
00[oo`03003ooooo08;oo`03003ooooo08;oo`03003ooooo00Woo`002_oo00<00?ooool0P_oo00<0
0?ooool0P_oo00<00?ooool02Ooo000:ool00`00ooooo`22ool00`00ooooo`22ool00`00ooooo`09
ool000[oo`03003ooooo08;oo`03003ooooo08;oo`03003ooooo00Woo`002_oo0P00Pooo00<00?oo
ool0POoo00<00?ooool02_oo000;ool00`00ooooo`21ool00`00ooooo`21ool00`00ooooo`0:ool0
00_oo`03003ooooo087oo`03003ooooo087oo`03003ooooo00[oo`002ooo00<00?ooool0POoo00<0
0?ooool0POoo00<00?ooool02_oo000;ool20022ool00`00ooooo`20ool00`00ooooo`0;ool000co
o`03003ooooo083oo`03003ooooo083oo`03003ooooo00_oo`003?oo00<00?ooool0P?oo00<00?oo
ool0P?oo00<00?ooool02ooo000<ool20021ool00`00ooooo`1oool00`00ooooo`0<ool000coo`80
087oo`03003ooooo07ooo`03003ooooo00coo`003Ooo00<00?ooool0Oooo0P00P?oo00<00?ooool0
3?oo000=ool00`00ooooo`1oool00`00ooooo`1oool00`00ooooo`0<ool000goo`80083oo`03003o
oooo07koo`03003ooooo00goo`003Ooo0P00P?oo00<00?ooool0O_oo00<00?ooool03Ooo000>ool0
0`00ooooo`1nool00`00ooooo`1nool00`00ooooo`0=ool000koo`03003ooooo07koo`03003ooooo
07koo`03003ooooo00goo`003ooo00<00?ooool0OOoo00<00?ooool0OOoo00<00?ooool03_oo000?
ool00`00ooooo`1mool00`00ooooo`1mool00`00ooooo`0>ool000ooo`8007koo`03003ooooo07co
o`03003ooooo00ooo`004?oo00<00?ooool0O?oo00<00?ooool0O?oo00<00?ooool03ooo000@ool2
001mool00`00ooooo`1kool00`00ooooo`0@ool0017oo`03003ooooo07_oo`03003ooooo07_oo`03
003ooooo013oo`004Ooo0P00O?oo00<00?ooool0N_oo00<00?ooool04Ooo000Bool00`00ooooo`1j
ool00`00ooooo`1jool00`00ooooo`0Aool001;oo`8007_oo`8007[oo`03003ooooo01;oo`004ooo
00<00?ooool0NOoo00<00?ooool0NOoo00<00?ooool04_oo000Cool00`00ooooo`1iool00`00oooo
o`1hool00`00ooooo`0Cool001Coo`03003ooooo07Soo`03003ooooo07Soo`03003ooooo01?oo`00
5?oo00<00?ooool0N?oo00<00?ooool0Mooo00<00?ooool05?oo000Eool00`00ooooo`1gool00`00
ooooo`1gool00`00ooooo`0Dool001Goo`03003ooooo07Ooo`03003ooooo07Koo`03003ooooo01Go
o`005_oo00<00?ooool0M_oo00<00?ooool0M_oo00<00?ooool05Ooo000Fool00`00ooooo`1fool0
0`00ooooo`1eool00`00ooooo`0Fool001Ooo`03003ooooo07Goo`03003ooooo07Goo`03003ooooo
01Koo`005ooo00<00?ooool0MOoo00<00?ooool0M?oo00<00?ooool05ooo000Hool00`00ooooo`1P
ool20004ool00`00ooooo`02ool30006ool00`00ooooo`1dool00`00ooooo`0Gool001Soo`03003o
oooo05ooo`04003ooooo000;ool00`00ooooo`03ool00`00ooooo`1cool00`00ooooo`0Hool001Wo
o`03003ooooo05koo`04003ooooo000;ool00`00ooooo`03ool3001cool00`00ooooo`0Hool001Wo
o`03003ooooo05koo`04003ooooo0008ool30006ool00`00ooooo`1bool00`00ooooo`0Iool001[o
o`03003ooooo05goo`04003ooooo0008ool00`00ooooo`06ool00`00ooooo`1bool00`00ooooo`0I
ool001[oo`03003ooooo05goo`04003ooooo0008ool00`00ooooo`06ool00`00ooooo`1aool00`00
ooooo`0Jool001_oo`03003ooooo05goo`8000Woo`@000Goo`03003ooooo077oo`03003ooooo01[o
o`006ooo0P00L_oo00<00?ooool0L?oo00<00?ooool06ooo000Lool2001aool00`00ooooo`1_ool0
0`00ooooo`0Lool001goo`03003ooooo06ooo`03003ooooo06koo`03003ooooo01goo`007_oo00<0
0?ooool0K_oo00<00?ooool0K_oo00<00?ooool07Ooo000Oool00`00ooooo`1]ool00`00ooooo`1]
ool00`00ooooo`0Nool0023oo`03003ooooo06coo`03003ooooo06coo`03003ooooo01ooo`008?oo
00<00?ooool0K?oo00<00?ooool0Jooo00<00?ooool08?oo000Qool00`00ooooo`1[ool00`00oooo
o`1[ool00`00ooooo`0Pool002;oo`03003ooooo06[oo`03003ooooo06[oo`03003ooooo027oo`00
8_oo00<00?ooool0J_oo0P00J_oo00<00?ooool08_oo000Sool00`00ooooo`1Yool00`00ooooo`1X
ool00`00ooooo`0Sool002Coo`03003ooooo06Soo`03003ooooo06Ooo`03003ooooo02Coo`009Ooo
00<00?ooool0Iooo00<00?ooool0Iooo00<00?ooool09?oo000Uool00`00ooooo`1Wool00`00oooo
o`1Vool00`00ooooo`0Uool002Koo`03003ooooo06Koo`03003ooooo06Goo`03003ooooo02Koo`00
9ooo00<00?ooool0IOoo00<00?ooool0I?oo00<00?ooool09ooo000Xool00`00ooooo`1Tool00`00
ooooo`1Sool00`00ooooo`0Xool002Soo`03003ooooo06Coo`03003ooooo06?oo`03003ooooo02So
o`00:Ooo00<00?ooool0Hooo00<00?ooool0H_oo00<00?ooool0:Ooo000Zool00`00ooooo`1Rool0
0`00ooooo`1Qool00`00ooooo`0Zool002_oo`03003ooooo067oo`03003ooooo063oo`03003ooooo
02_oo`00;?oo0P00HOoo00<00?ooool0H?oo00<00?ooool0:ooo000^ool00`00ooooo`1Nool00`00
ooooo`1Oool00`00ooooo`0/ool002ooo`03003ooooo05goo`8005ooo`03003ooooo02goo`00<?oo
00<00?ooool0G?oo00<00?ooool0GOoo00<00?ooool0;_oo000aool00`00ooooo`1Kool00`00oooo
o`1Kool2000aool003;oo`03003ooooo05[oo`03003ooooo05[oo`03003ooooo037oo`00<ooo00<0
0?ooool0FOoo00<00?ooool0FOoo00<00?ooool0<_oo000dool2001Iool00`00ooooo`1Hool00`00
ooooo`0cool003Koo`03003ooooo05Koo`03003ooooo05Koo`8003Koo`00=ooo00<00?ooool0EOoo
00<00?ooool0EOoo00<00?ooool0=_oo000hool00`00ooooo`1Dool00`00ooooo`1Dool00`00oooo
o`0gool003Woo`03003ooooo05?oo`03003ooooo05?oo`03003ooooo03Soo`00>_oo00<00?ooool0
D_oo00<00?ooool0D_oo00<00?ooool0>Ooo000kool2001Bool00`00ooooo`1@ool2000lool003go
o`03003ooooo04ooo`03003ooooo04ooo`03003ooooo03coo`00?_oo00<00?ooool0C_oo0P00Cooo
00<00?ooool0?Ooo000oool00`00ooooo`1=ool00`00ooooo`1=ool00`00ooooo`0nool0043oo`80
04goo`03003ooooo04_oo`80047oo`00@_oo00<00?ooool0B_oo00<00?ooool0B_oo00<00?ooool0
@Ooo0013ool2001:ool00`00ooooo`19ool00`00ooooo`12ool004Goo`8004Soo`03003ooooo04Oo
o`8004Goo`00Aooo00<00?ooool0AOoo00<00?ooool0AOoo0P00Aooo0018ool20015ool00`00oooo
o`13ool20019ool004[oo`03003ooooo04;oo`03003ooooo047oo`8004_oo`00Booo0P00@_oo00<0
0?ooool0@?oo00<00?ooool0Booo001=ool20010ool00`00ooooo`0nool2001>ool004ooo`03003o
oooo03goo`03003ooooo03coo`80053oo`00D?oo0P00?Ooo00<00?ooool0>_oo0P00D_oo001Bool0
0`00ooooo`0jool00`00ooooo`0iool00`00ooooo`1Bool005?oo`8003[oo`8003Soo`8005Goo`00
EOoo0P00>?oo00<00?ooool0=Ooo0P00Eooo001Gool2000fool00`00ooooo`0cool2001Iool005Wo
o`<003?oo`03003ooooo037oo`8005_oo`00G?oo1000;ooo00<00?ooool0;ooo0P00GOoo001Pool3
000/ool00`00ooooo`0[ool4001Oool006?oo`<002Woo`03003ooooo02Soo`<006?oo`00I_oo0`00
9_oo00<00?ooool09?oo1000I_oo001Yool4000Rool00`00ooooo`0Pool4001Zool006goo`<001oo
o`03003ooooo01goo`<006koo`00L?oo10006ooo00<00?ooool06Ooo1000LOoo001dool4000Gool0
0`00ooooo`0Dool5001eool007Soo`D000Woo`@000Goo`03003ooooo00goo`L007[oo`00OOoo3000
1_oo00<00?ooool01_oo1`00POoo0028ool@0028ool008Soo`03003ooooo00Coo`03003ooooo08ko
o`00R?oo00<00?ooool01?oo00<00?ooool0S_oo0026ool30006ool00`00ooooo`2>ool008Soo`03
003ooooo00Coo`03003ooooo08koo`00Sooo00<00?ooool0S_oo002?ool00`00ooooo`2>ool00001
\
\>"],
  ImageRangeCache->{{{0, 287}, {287, 0}} -> {-1.05001, -1.05001, 0.00731714, \
0.00731714}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[6]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Global versus local", "Subsection"],

Cell[TextData[{
  "The method ",
  StyleBox["LocallyExact", "MR"],
  " is not intended as a substitute for a closed-form (global) solution."
}], "Text"],

Cell[TextData[{
  "Despite the fact that the method ",
  StyleBox["LocallyExact", "MR"],
  " uses the analytic solution to advance the solution, it only produces \
solutions at the grid points in the numerical integration (or even inside \
grid points if called appropriately). Therefore, there can be errors due to \
sampling at interpolation points that do not lie exactly on the numerical \
integration grid."
}], "Text"],

Cell["\<\
Plot the error in the first solution component of the harmonic \
oscillator and compare it with the exact flow.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Plot[Evaluate[First[sols] - Cos[T]], Evaluate[tdata]]\)], "Input",
  CellLabel->"In[7]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.0952381 0.309152 1.13301e+06 [
[.21429 .29665 -3 -9 ]
[.21429 .29665 3 0 ]
[.40476 .29665 -3 -9 ]
[.40476 .29665 3 0 ]
[.59524 .29665 -3 -9 ]
[.59524 .29665 3 0 ]
[.78571 .29665 -3 -9 ]
[.78571 .29665 3 0 ]
[.97619 .29665 -6 -9 ]
[.97619 .29665 6 0 ]
[.01131 .08255 -39 -6.25 ]
[.01131 .08255 0 6.25 ]
[.01131 .19585 -39 -6.25 ]
[.01131 .19585 0 6.25 ]
[.01131 .42245 -33 -6.25 ]
[.01131 .42245 0 6.25 ]
[.01131 .53575 -33 -6.25 ]
[.01131 .53575 0 6.25 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .30915 m
.21429 .3154 L
s
[(2)] .21429 .29665 0 1 Mshowa
.40476 .30915 m
.40476 .3154 L
s
[(4)] .40476 .29665 0 1 Mshowa
.59524 .30915 m
.59524 .3154 L
s
[(6)] .59524 .29665 0 1 Mshowa
.78571 .30915 m
.78571 .3154 L
s
[(8)] .78571 .29665 0 1 Mshowa
.97619 .30915 m
.97619 .3154 L
s
[(10)] .97619 .29665 0 1 Mshowa
.125 Mabswid
.07143 .30915 m
.07143 .3129 L
s
.11905 .30915 m
.11905 .3129 L
s
.16667 .30915 m
.16667 .3129 L
s
.2619 .30915 m
.2619 .3129 L
s
.30952 .30915 m
.30952 .3129 L
s
.35714 .30915 m
.35714 .3129 L
s
.45238 .30915 m
.45238 .3129 L
s
.5 .30915 m
.5 .3129 L
s
.54762 .30915 m
.54762 .3129 L
s
.64286 .30915 m
.64286 .3129 L
s
.69048 .30915 m
.69048 .3129 L
s
.7381 .30915 m
.7381 .3129 L
s
.83333 .30915 m
.83333 .3129 L
s
.88095 .30915 m
.88095 .3129 L
s
.92857 .30915 m
.92857 .3129 L
s
.25 Mabswid
0 .30915 m
1 .30915 L
s
.02381 .08255 m
.03006 .08255 L
s
gsave
.01131 .08255 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .19585 m
.03006 .19585 L
s
gsave
.01131 .19585 -39 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(-1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .42245 m
.03006 .42245 L
s
gsave
.01131 .42245 -33 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.02381 .53575 m
.03006 .53575 L
s
gsave
.01131 .53575 -33 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-7) show
grestore
.125 Mabswid
.02381 .10521 m
.02756 .10521 L
s
.02381 .12787 m
.02756 .12787 L
s
.02381 .15053 m
.02756 .15053 L
s
.02381 .17319 m
.02756 .17319 L
s
.02381 .21851 m
.02756 .21851 L
s
.02381 .24117 m
.02756 .24117 L
s
.02381 .26383 m
.02756 .26383 L
s
.02381 .28649 m
.02756 .28649 L
s
.02381 .33181 m
.02756 .33181 L
s
.02381 .35447 m
.02756 .35447 L
s
.02381 .37713 m
.02756 .37713 L
s
.02381 .39979 m
.02756 .39979 L
s
.02381 .44511 m
.02756 .44511 L
s
.02381 .46777 m
.02756 .46777 L
s
.02381 .49043 m
.02756 .49043 L
s
.02381 .51309 m
.02756 .51309 L
s
.02381 .05989 m
.02756 .05989 L
s
.02381 .03723 m
.02756 .03723 L
s
.02381 .01457 m
.02756 .01457 L
s
.02381 .55841 m
.02756 .55841 L
s
.02381 .58107 m
.02756 .58107 L
s
.02381 .60374 m
.02756 .60374 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .30915 m
.02498 .25452 L
.02615 .1475 L
.02731 .0541 L
.02848 .01472 L
.02965 .04408 L
.03082 .13128 L
.03199 .23982 L
.03316 .30759 L
.03432 .2685 L
.03549 .16552 L
.03666 .06783 L
.03783 .01924 L
.039 .03814 L
.04017 .11754 L
.04134 .2251 L
.0425 .30319 L
.04377 .27473 L
.04504 .16651 L
.0463 .06512 L
.04757 .02337 L
.04884 .05964 L
.0501 .15794 L
.05137 .26807 L
.05264 .30611 L
.0539 .22899 L
.05517 .11872 L
.05644 .04395 L
.0577 .03977 L
.05897 .1079 L
.06024 .21682 L
.0615 .30194 L
.06277 .28005 L
.06395 .18774 L
.06514 .09529 L
.06632 .04626 L
.0675 .05977 L
.06868 .13063 L
.06987 .22948 L
.07105 .30295 L
.07223 .28505 L
.07341 .19918 L
.0746 .11042 L
.07578 .06125 L
.07696 .07093 L
.07815 .1357 L
.07933 .22893 L
.08051 .30132 L
.08169 .28969 L
.08285 .21286 L
Mistroke
.08401 .12999 L
.08517 .08016 L
.08633 .08234 L
.08749 .13562 L
.08865 .21934 L
.08981 .29334 L
.09097 .30076 L
.09213 .23769 L
.09329 .15904 L
.09445 .10471 L
.09561 .09604 L
.09677 .13602 L
.09793 .20949 L
.09909 .28335 L
.10025 .30717 L
.1015 .25369 L
.10276 .17651 L
.10402 .12349 L
.10528 .11921 L
.10653 .16528 L
.10779 .24071 L
.10905 .30226 L
.11031 .29331 L
.11157 .23066 L
.11282 .16779 L
.11408 .13808 L
.11534 .15441 L
.1166 .20954 L
.11785 .27652 L
.11911 .30904 L
.12037 .27515 L
.12154 .22058 L
.12272 .17687 L
.12389 .1625 L
.12507 .18266 L
.12624 .22956 L
.12741 .2827 L
.12859 .30915 L
.12976 .28574 L
.13093 .24129 L
.13211 .20346 L
.13328 .18864 L
.13446 .20224 L
.13563 .23895 L
.1368 .2831 L
.13798 .30887 L
.13915 .29434 L
.14042 .25759 L
.1417 .22626 L
.14297 .2163 L
Mistroke
.14424 .23184 L
.14551 .26572 L
.14678 .29987 L
.14806 .30709 L
.14933 .28609 L
.1506 .26055 L
.15187 .24571 L
.15314 .24827 L
.15442 .26683 L
.15569 .29239 L
.15696 .30881 L
.15823 .30293 L
.15951 .28854 L
.16075 .277 L
.162 .27369 L
.16325 .27977 L
.1645 .29239 L
.16575 .30514 L
.167 .30899 L
.16825 .30667 L
.1695 .30404 L
.17075 .30286 L
.17199 .30353 L
.17324 .30549 L
.17449 .3077 L
.17574 .30904 L
.17699 .31113 L
.17824 .31904 L
.17949 .32755 L
.18881 .34884 L
.18997 .36027 L
.19114 .36011 L
.1923 .3482 L
.19347 .32908 L
.19463 .31227 L
.1958 .31301 L
.19696 .33713 L
.19813 .36647 L
.19939 .38747 L
.20065 .38778 L
.20192 .36702 L
.20318 .33466 L
.20444 .31055 L
.20571 .32285 L
.20697 .36386 L
.20823 .40229 L
.2095 .41834 L
.21076 .40474 L
.21202 .3672 L
.21329 .32484 L
Mistroke
.21455 .31071 L
.21581 .34812 L
.21708 .40183 L
.21834 .43886 L
.21952 .44277 L
.2207 .41512 L
.22188 .36668 L
.22306 .32084 L
.22424 .31383 L
.22542 .35945 L
.2266 .41911 L
.22777 .46133 L
.22895 .46874 L
.23013 .43838 L
.23131 .38196 L
.23249 .32611 L
.23367 .31259 L
.23485 .3619 L
.23603 .43066 L
.23721 .48147 L
.23837 .4934 L
.23952 .46327 L
.24068 .4021 L
.24183 .33668 L
.24299 .30977 L
.24415 .35425 L
.2453 .42934 L
.24646 .49211 L
.24761 .51645 L
.24877 .49336 L
.24993 .4311 L
.25108 .35544 L
.25224 .30987 L
.2534 .34186 L
.25455 .42121 L
.25571 .49649 L
.25696 .53565 L
.25822 .51425 L
.25947 .44128 L
.26073 .35223 L
.26198 .30939 L
.26323 .36538 L
.26449 .46179 L
.26574 .53564 L
.267 .55178 L
.26825 .50321 L
.26951 .41127 L
.27076 .32592 L
.27201 .32297 L
.27327 .40997 L
Mistroke
.27452 .50902 L
.27578 .56522 L
.27695 .55642 L
.27812 .49131 L
.27929 .39589 L
.28046 .31921 L
.28163 .32815 L
.2828 .41561 L
.28397 .51334 L
.28514 .57387 L
.28631 .57343 L
.28748 .51212 L
.28865 .41394 L
.28982 .32699 L
.29099 .32107 L
.29216 .40512 L
.29333 .50841 L
.29451 .57836 L
.29577 .58468 L
.29704 .51659 L
.29831 .40551 L
.29958 .31719 L
.30085 .33966 L
.30212 .44756 L
.30339 .55232 L
.30466 .59846 L
.30592 .56541 L
.30719 .46764 L
.30846 .35472 L
.30973 .31126 L
.311 .38964 L
.31227 .5068 L
.31354 .58852 L
.31481 .596 L
.31605 .52782 L
.3173 .41507 L
.31854 .32105 L
.31979 .33338 L
.32103 .43766 L
.32228 .54631 L
.32352 .60203 L
.32477 .58065 L
.32602 .49114 L
.32726 .37564 L
.32851 .30937 L
.32975 .36451 L
.331 .47834 L
.33224 .57254 L
.33349 .60179 L
.33473 .5538 L
Mistroke
.3359 .45711 L
.33706 .35338 L
.33822 .30991 L
.33938 .37216 L
.34054 .47762 L
.3417 .56502 L
.34287 .59777 L
.34403 .56387 L
.34519 .47584 L
.34635 .37066 L
.34751 .30973 L
.34867 .35283 L
.34983 .45272 L
.351 .54493 L
.35216 .58905 L
.35332 .56866 L
.35458 .48293 L
.35584 .37178 L
.3571 .30925 L
.35836 .36307 L
.35962 .46983 L
.36088 .55557 L
.36214 .57853 L
.3634 .52893 L
.36466 .4288 L
.36592 .33178 L
.36718 .31985 L
.36844 .40399 L
.3697 .50366 L
.37096 .56197 L
.37222 .55266 L
.37348 .47984 L
.37465 .38438 L
.37583 .31461 L
.37701 .333 L
.37818 .41546 L
.37936 .50039 L
.38053 .54763 L
.38171 .53889 L
.38289 .47756 L
.38406 .3885 L
.38524 .31787 L
.38641 .32583 L
.38759 .3992 L
.38876 .47957 L
.38994 .52776 L
.39112 .5249 L
.39229 .47219 L
.39357 .38377 L
.39484 .31483 L
Mistroke
.39612 .33214 L
.39739 .40941 L
.39866 .48258 L
.39994 .51286 L
.40121 .48687 L
.40249 .41636 L
.40376 .33777 L
.40503 .31143 L
.40631 .36335 L
.40758 .43598 L
.40886 .48386 L
.41013 .48443 L
.41141 .43767 L
.41268 .3657 L
.41393 .31284 L
.41518 .32768 L
.41643 .38658 L
.41768 .44232 L
.41894 .46634 L
.42019 .44845 L
.42144 .3965 L
.42269 .33598 L
.42394 .30954 L
.42519 .34197 L
.42644 .39411 L
.42769 .43239 L
.42894 .43895 L
.43019 .41125 L
.43145 .3617 L
.4327 .31721 L
.43386 .31388 L
.43503 .34565 L
.4362 .38393 L
.43736 .40927 L
.43853 .41174 L
.4397 .39063 L
.44087 .35421 L
.44203 .31938 L
.4432 .31068 L
.44437 .3315 L
.44553 .35996 L
.4467 .3806 L
.44787 .38511 L
.44904 .37203 L
.4502 .34649 L
.45137 .31986 L
.45263 .3097 L
.4539 .32344 L
.45517 .34265 L
.45643 .35516 L
Mistroke
.4577 .35524 L
.45896 .34313 L
.46023 .32459 L
.46149 .31039 L
.46276 .31137 L
.46402 .31874 L
.46529 .32531 L
.46655 .32759 L
.46782 .32482 L
.46908 .31846 L
.47035 .31175 L
.47162 .3091 L
.4728 .30697 L
.47398 .30317 L
.47516 .29965 L
.47634 .29799 L
.47752 .29903 L
.4787 .30257 L
.47989 .30703 L
.48107 .30907 L
.48225 .30068 L
.48343 .286 L
.48461 .27352 L
.48579 .26866 L
.48697 .27348 L
.48816 .28634 L
.48934 .30157 L
.49052 .30913 L
.49168 .29609 L
.49283 .27139 L
.49399 .24954 L
.49515 .23981 L
.49631 .24598 L
.49747 .266 L
.49863 .29171 L
.49978 .30858 L
.50094 .29702 L
.5021 .2637 L
.50326 .23066 L
.50442 .21266 L
.50557 .2166 L
.50673 .24126 L
.50789 .27702 L
.50905 .30556 L
.51031 .29799 L
.51156 .25323 L
.51282 .20741 L
.51408 .18474 L
.51533 .19538 L
.51659 .23501 L
Mistroke
.51784 .28441 L
.5191 .30908 L
.52036 .27554 L
.52161 .21619 L
.52287 .17006 L
.52413 .15926 L
.52538 .18864 L
.52664 .24536 L
.5279 .29853 L
.52915 .30001 L
.53033 .24624 L
.5315 .18248 L
.53267 .14059 L
.53384 .1373 L
.53502 .17402 L
.53619 .23654 L
.53736 .29478 L
.53853 .30309 L
.53971 .2469 L
.54088 .17395 L
.54205 .1223 L
.54322 .11294 L
.5444 .14949 L
.54557 .21802 L
.54674 .28675 L
.54791 .30601 L
.54919 .24335 L
.55046 .1563 L
.55173 .09841 L
.553 .09699 L
.55427 .15286 L
.55554 .23997 L
.55681 .30513 L
.55808 .27933 L
.55935 .18888 L
.56062 .10497 L
.56189 .07134 L
.56316 .10305 L
.56444 .1862 L
.56571 .27767 L
.56698 .30514 L
.56825 .23155 L
.57823 .18826 L
.57948 .0907 L
.58072 .04269 L
.58197 .06512 L
.58322 .14864 L
.58447 .25346 L
.58571 .30915 L
.58696 .25149 L
Mistroke
.58821 .14265 L
.58937 .05967 L
.59054 .03045 L
.5917 .06581 L
.59286 .15271 L
.59403 .25415 L
.59519 .30905 L
.59636 .25999 L
.59752 .15698 L
.59868 .06453 L
.59985 .02275 L
.60101 .04726 L
.60217 .12912 L
.60334 .23475 L
.6045 .30589 L
.60567 .27472 L
.60683 .17418 L
.60809 .06742 L
.60935 .01753 L
.61061 .04656 L
.61188 .14199 L
.61314 .2566 L
.6144 .30848 L
.61566 .23702 L
.61692 .11991 L
.61819 .03329 L
.61945 .01846 L
.62071 .08181 L
.62197 .19479 L
.62323 .29396 L
.6245 .28805 L
.62576 .18435 L
.62702 .0739 L
.6282 .01839 L
.62938 .03142 L
.63055 .1081 L
.63173 .2171 L
.63291 .30068 L
.63409 .28431 L
.63527 .18721 L
.63644 .08382 L
.63762 .0239 L
.6388 .03094 L
.63998 .10228 L
.64116 .20914 L
.64233 .29666 L
.64351 .2904 L
.64469 .19893 L
.64587 .09606 L
.64702 .03332 L
Mistroke
.64818 .03248 L
.64933 .09378 L
.65049 .19389 L
.65164 .28594 L
.6528 .30141 L
.65395 .22528 L
.65511 .1249 L
.65626 .0521 L
.65742 .03566 L
.65857 .08141 L
.65973 .17236 L
.66088 .26877 L
.66204 .30836 L
.66319 .2514 L
.66435 .1556 L
.6656 .07125 L
.66685 .04656 L
.66811 .09195 L
.66936 .18753 L
.67061 .28334 L
.67186 .3015 L
.67312 .22402 L
.67437 .1272 L
.67562 .06709 L
.67688 .07075 L
.67813 .13648 L
.67938 .23409 L
.68064 .30508 L
.68189 .27977 L
.68314 .19233 L
.6844 .11108 L
.68557 .07774 L
.68673 .09848 L
.6879 .16549 L
.68907 .25091 L
.69024 .30703 L
.69141 .28232 L
.69258 .20793 L
.69375 .13569 L
.69492 .09849 L
.69609 .11036 L
.69726 .16679 L
.69843 .24488 L
.6996 .30357 L
.70076 .29234 L
.70193 .22862 L
.7031 .16116 L
.70437 .1206 L
.70564 .13129 L
.7069 .18837 L
Mistroke
.70817 .26424 L
.70944 .30892 L
.71071 .27706 L
.71197 .21055 L
.71324 .15683 L
.71451 .14246 L
.71578 .17349 L
.71704 .23583 L
.71831 .2957 L
.71958 .30271 L
.72085 .25503 L
.72211 .19981 L
.72338 .16875 L
.72457 .17417 L
.72575 .21076 L
.72693 .26343 L
.72812 .3045 L
.7293 .29972 L
.73048 .26074 L
.73167 .21914 L
.73285 .1953 L
.74232 .22279 L
.76088 .28071 L
.78102 .33746 L
.79042 .36656 L
.79159 .36108 L
.79277 .34392 L
.79394 .32234 L
.79512 .30931 L
.79629 .32212 L
.79747 .35258 L
.79864 .38106 L
.79981 .39484 L
.80109 .38704 L
.80236 .35891 L
.80363 .32455 L
.80491 .30955 L
.80618 .33788 L
.80745 .38317 L
.80873 .41621 L
.81 .42097 L
.81127 .39512 L
.81254 .35043 L
.81382 .31323 L
.81509 .32237 L
.81636 .37379 L
.81764 .42531 L
.81891 .44924 L
.82018 .43464 L
.82143 .38874 L
Mistroke
.82268 .33401 L
.82393 .30956 L
.82518 .34895 L
.82643 .41385 L
.82768 .46292 L
.82893 .47307 L
.83018 .4398 L
.83143 .37756 L
.83268 .32015 L
.83393 .31943 L
.83518 .38243 L
.83643 .45415 L
.83768 .4952 L
.83893 .4873 L
.84018 .43356 L
.84134 .36371 L
.84251 .31296 L
.84367 .32974 L
.84484 .4003 L
.84601 .4738 L
.84717 .51606 L
.84834 .51077 L
.8495 .45969 L
.85067 .38293 L
.85183 .31909 L
.853 .3227 L
.85417 .39424 L
.85533 .4772 L
.8565 .53089 L
.85766 .53422 L
.85883 .48583 L
.86009 .397 L
.86136 .32003 L
.86262 .32824 L
.86389 .41626 L
.86515 .50865 L
.86641 .55528 L
.86768 .53523 L
.86894 .45705 L
.87021 .35898 L
.87147 .30923 L
.87273 .36768 L
.874 .47172 L
.87526 .55173 L
.87653 .56902 L
.87779 .51598 L
.87905 .41636 L
.88023 .32962 L
.88141 .31849 L
.88259 .39839 L
Mistroke
.88378 .49987 L
.88496 .56974 L
.88614 .5795 L
.88732 .5254 L
.8885 .4286 L
.88968 .33527 L
.89086 .31579 L
.89204 .39419 L
.89322 .4997 L
.8944 .57548 L
.89558 .59029 L
.89676 .5385 L
.89794 .44022 L
.89909 .34293 L
.90025 .31219 L
.90141 .38277 L
.90256 .48894 L
.90372 .57247 L
.90488 .59933 L
.90604 .55973 L
.90719 .4682 L
.90835 .36359 L
.90951 .30917 L
.91066 .36129 L
.91182 .46663 L
.91298 .56049 L
.91413 .60273 L
.91529 .57777 L
.91645 .49465 L
.9177 .37843 L
.91896 .30956 L
.92021 .36357 L
.92147 .47872 L
.92272 .57419 L
.92398 .60332 L
.92523 .55364 L
.92649 .44686 L
.92774 .33883 L
.929 .31779 L
.93025 .40837 L
.93151 .52239 L
.93276 .59361 L
.93402 .58975 L
.93528 .51248 L
.93653 .39736 L
.9377 .31649 L
.93887 .33508 L
.94004 .43121 L
.94122 .5326 L
.94239 .59119 L
Mistroke
.94356 .58429 L
.94473 .51449 L
.9459 .40961 L
.94707 .32261 L
.94824 .32597 L
.94941 .41404 L
.95059 .51514 L
.95176 .5793 L
.95293 .58125 L
.9541 .52036 L
.95527 .42048 L
.95654 .32482 L
.95781 .32746 L
.95908 .42228 L
.96035 .52389 L
.96162 .57602 L
.96289 .55523 L
.96416 .47071 L
.96543 .36408 L
.9667 .30919 L
.96797 .36609 L
.96924 .46804 L
.97051 .54583 L
.97177 .5616 L
.97304 .50861 L
.97431 .41102 L
.97558 .32362 L
.97619 .30915 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[7]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00Ooooo8Ooo003ooolQ
ool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00
oooo8Ooo003ooolQool002[oo`03003ooooo0??oo`00:_oo00<00?ooool0looo000Zool00`00oooo
o`3cool002[oo`800?Coo`00:_oo0P0000?oo`00ool0SOoo00H00?ooool00?oo001Nool002[oo`80
0005ool00?ooool00029ool00`00ool00002ool00`00ool0001Nool002[oo`800005ool00?ooool0
0029ool00`00ool00002ool01P00ool00000ool005_oo`00:_oo0P0000Goo`00ooooo`0008Ooo`05
003oo`00ool00002ool01P00ool00000ool005_oo`00:_oo0P0000Goo`00ooooo`0008Ooo`04003o
o`00ool200000ooo003oo`0200001?oo003oo`000_oo00<00?ooool0E?oo000Zool200001Ooo003o
oooo0000Qooo00@00?oo003oo`800003ool00?oo00800004ool00?oo0002ool00`00ooooo`1Dool0
02[oo`800007ool00?ooool00?oo0000P_oo00L00?ooool00?oo003oo`0200000ooo003oo`020000
1?oo003oo`000_oo00<00?ooool0E?oo000Zool200000ooo003oo`0200001?oo003oo`00P?oo00L0
0?oo00000?oo003oo`0200000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool0Dooo000Z
ool200000ooo003oo`0200001?oo003oo`00P?oo00L00?oo00000?oo003oo`0200000ooo003oo`02
00001Ooo003oo`00ool00P0000?oo`00ool0Dooo000Zool200000ooo003oo`0200001?oo003oo`00
P?oo00L00?oo00000?oo003oo`0200000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool0
Dooo000Zool200000ooo003oo`0200000ooo003oo`02001oool400000ooo003oo`0200000ooo003o
o`0400000ooo003oo`0200000ooo003oo`1Cool002[oo`800003ool00?oo00800003ool00?oo0080
0003ool00?oo07coo`@00003ool00?oo00800003ool00?oo00@00003ool00?oo00800003ool00?oo
05?oo`00:_oo0P0000?oo`00ool00P0000?oo`00ool00P0000?oo`00ool0O?oo100000?oo`00ool0
0P0000?oo`00ool0100000?oo`0000000P0000Coo`00ool005;oo`001ooo10000_oo00D00?oooooo
o`0000Koo`@000?oo`8000Woo`800003ool00?oo00800003ool00?oo00800003ool00?oo07Soo`04
003oooooool400000ooo003oo`0200000ooo003oo`0400000ooo0000000200001?oo003oo`00D_oo
0007ool00`00ooooo`04ool00`00ool00009ool01@00oooooooo00000_oo00<00?ooool01_oo0P00
00?oo`00ool00P0000?oo`00ool00P0000?oo`00ool0N?oo00@00?ooooooo`@00003ool00?oo0080
0003ool00?oo00@00003ool0000000800004ool00?oo001Bool000Soo`03003ooooo00Coo`03003o
oooo00Soo`05003oooooool00002ool00`00ooooo`06ool200000ooo003oo`0200000ooo003oo`02
00000ooo003oo`1hool01000oooooooo100000?oo`00ool00P0000?oo`00ool0100000?oo`000000
0P0000Coo`00ool000;oo`03003ooooo04goo`0000?oo`0000000P001?oo00<00?ooool00_oo00<0
0?oo00002Ooo00D00?ooooooo`0000;oo`03003ooooo00Koo`800003ool0000000800003ool00?oo
00800004ool00?oo001gool01000oooooooo100000?oo`00ool0100000?oo`0000000P0000?oo`00
00000P0000Coo`00ool000;oo`03003ooooo04goo`002_oo00@00?ooool000?oo`03003ooooo00Ko
o`05003oooooool00002ool00`00ooooo`06ool200000ooo0000000200000ooo003oo`0200001?oo
003oo`00Mooo00@00?ooooooo`@00003ool00?oo00@00003ool0000000800003ool0000000800004
ool00?oo0002ool00`00ooooo`1=ool000Ooo`04003ooooo000=ool30003ool01000ooooo`002?oo
0P0000?oo`0000000P0000?oo`00ool0100000?oo`00ool0MOoo0P000ooo100000?oo`0000001000
00?oo`0000000P0000?oo`0000000P0000Coo`00ool000;oo`03003ooooo04goo`002?oo0P004?oo
00<00?ooool00_oo0P002Ooo0P0000?oo`0000000P0000?oo`00ool0100000?oo`00ool0MOoo0P00
0ooo100000?oo`000000100000?oo`0000000P0000?oo`0000000P0000Coo`00ool000;oo`03003o
oooo04goo`009?oo0`000ooo0P0000?oo`0000000P0000?oo`00ool0100000Goo`00ooooo`00077o
o`04003oo`000003ool400000ooo0000000400000ooo0000000200000ooo0000000200001Ooo003o
o`00ool00P0000?oo`00ool0C?oo000Zool200000ooo0000000200000ooo003oo`0400001Ooo003o
oooo0000LOoo00@00?oo000000?oo`@00003ool0000000@00003ool0000000800003ool000000080
0005ool00?oo003oo`0200000ooo003oo`1<ool002Woo`<00003ool0000000800003ool00?oo00@0
0003ool00?oo0080077oo`04003oo`000003ool400000ooo0000000400000ooo0000000200000ooo
0000000200000ooo003oo`0400000ooo003oo`1<ool002Woo`<00003ool0000000800003ool00?oo
00@00003ool00?oo0080077oo`04003oo`000003ool400000ooo0000000400000ooo000000020000
0ooo0000000200000ooo003oo`0400000ooo003oo`1<ool002[oo`800003ool0000000800003ool0
0?oo00@00003ool00000008006ooo`04003oo`00ool20003ool400000ooo0000000400000ooo0000
000200000ooo0000000200000ooo003oo`0400000ooo003oo`1<ool002[oo`800003ool000000080
0003ool00?oo00@00003ool0000000800003ool00?oo06coo`04003oo`00ool20003ool400000ooo
0000000400000ooo0000000200000ooo0000000400000ooo0000000200000ooo003oo`1<ool002[o
o`800003ool0000000800003ool00?oo00@00003ool0000000800003ool00?oo06coo`04003oo`00
ool20003ool400000ooo0000000400000ooo0000000200000ooo0000000400000ooo000000020000
1?oo003oo`00Booo000Zool200000ooo0000000200000ooo003oo`0400000ooo0000000200000ooo
003oo`1/ool01000ool00?oo0P000ooo100000?oo`000000100000?oo`0000000P0000?oo`000000
100000?oo`0000000P0000Coo`00ool004_oo`00:_oo0P0000?oo`0000000P0000?oo`00ool01000
00?oo`0000000P0000?oo`00ool0K?oo00<00?oo00000`000ooo100000?oo`000000100000?oo`00
00000P0000?oo`000000100000?oo`0000000P0000?oo`00ool00P00B_oo000Zool200000ooo0000
000200000ooo003oo`0400000ooo0000000200000ooo003oo`1/ool00`00ool000030003ool40000
0ooo0000000400000ooo0000000200000ooo0000000400000ooo0000000200000ooo003oo`02001:
ool002[oo`800003ool0000000800003ool00?oo00@00003ool0000000800004ool00?oo001Yool0
1000ool00?oo10000ooo100000?oo`000000100000?oo`0000000P0000?oo`000000100000?oo`00
00000P0000?oo`00ool00P00B_oo000Zool200000ooo0000000200000ooo003oo`0400000ooo0000
000200001?oo003oo`00J?oo0P0000?oo`00ool010000ooo100000?oo`0000000`0000?oo`000000
0`0000?oo`000000100000?oo`0000000P0000?oo`0000000P0000?oo`00ool0Aooo000Zool20000
0ooo0000000200000ooo003oo`0400000ooo0000000200001?oo003oo`00J?oo0P0000?oo`00ool0
10000ooo100000?oo`0000000`0000?oo`0000000`0000?oo`0000000`0000?oo`0000000`0000?o
o`0000000P0000?oo`00ool0Aooo000Zool200000ooo0000000200000ooo003oo`0400000ooo0000
000200000ooo003oo`02001Wool400000ooo000000020003ool400000ooo0000000300000ooo0000
000400000ooo0000000200000ooo0000000300000ooo0000000200000ooo003oo`17ool002[oo`80
0003ool0000000800003ool00?oo00@00003ool0000000@00003ool0000006Ooo`@00003ool00000
008000?oo`@00003ool0000000<00003ool0000000@00003ool0000000800003ool0000000<00003
ool0000000800003ool00?oo04Ooo`00:_oo0P0000?oo`0000000P0000?oo`00ool0100000?oo`00
0000100000?oo`000000IOoo00<00?oo00000`0000?oo`0000000P000ooo100000?oo`0000000`00
00?oo`000000100000?oo`0000000P0000?oo`0000000`0000?oo`0000000P0000?oo`00ool0Aooo
000Zool200000ooo0000000200000ooo003oo`0400000ooo0000000400001Ooo00000?oo0000Hooo
00<00?oo00000`0000?oo`0000000P000ooo100000?oo`0000000`0000?oo`000000100000?oo`00
00000P0000?oo`0000000`0000?oo`0000000P0000?oo`00ool0Aooo000Zool200000ooo00000002
00000ooo0000000400000ooo0000000400001Ooo00000?oo0000Hooo00<00?oo00000`0000?oo`00
00000P000ooo100000?oo`0000000`0000?oo`000000100000?oo`0000000P0000?oo`0000000`00
00?oo`0000000P0000?oo`00ool0Aooo000Zool200000ooo0000000200000ooo0000000400000ooo
0000000400001Ooo00000?oo0000Hooo00<00?oo00000`0000?oo`0000000P000ooo100000?oo`00
00000`0000?oo`000000100000?oo`0000000P0000?oo`0000000`0000?oo`0000000P0000Coo`00
ool004Koo`00:_oo0P0000?oo`0000000P0000?oo`0000001@0000?oo`0000000`0000?oo`000000
0P00Hooo00<00?oo00000`0000?oo`0000000P000ooo100000?oo`0000000`0000?oo`0000001000
00?oo`0000000P0000?oo`0000000`0000?oo`0000000P0000Coo`00ool004Koo`00:_oo0P0000?o
o`0000000`0000?oo`000000100000?oo`0000000`0000?oo`0000000P00Hooo00<00?oo00000`00
00?oo`0000000P000ooo100000?oo`0000000`0000?oo`000000100000?oo`0000000P0000?oo`00
ool0100000Soo`0000000?oo003oo`00A_oo0007ool40002ool01@00oooooooo00001_oo10000ooo
0P002Ooo0P0000?oo`0000000`0000?oo`000000100000?oo`0000000`0000?oo`0000000P00Hooo
00<00?oo00000`0000?oo`0000000P000ooo100000?oo`0000000`0000?oo`000000100000?oo`00
00000P0000?oo`00ool0100000Soo`0000000?oo003oo`00A_oo0009ool00`00ooooo`02ool00`00
ool00009ool01@00oooooooo00000_oo00<00?ooool01_oo0P0000?oo`0000000`0000?oo`000000
0P0000?oo`00ool01@0000?oo`0000000P00HOoo00@00?oo003oo`@00003ool00000008000?oo`@0
0003ool0000000<00003ool0000000@00003ool0000000800003ool00?oo00@00003ool0000000<0
0003ool00?oo04Goo`002Ooo00<00?ooool00ooo00<00?ooool02?oo00D00?ooooooo`0000;oo`03
003ooooo00Koo`03003oo`0000D00003ool0000000800003ool00?oo00D00003ool0000000800003
ool00?oo05koo`04003oo`00ool400000ooo000000020003ool400000ooo0000000300000ooo0000
000400000ooo0000000200000ooo003oo`0400000ooo0000000300000ooo003oo`15ool00003ool0
0000008000Coo`03003ooooo00;oo`03003oo`0000Woo`05003oooooool00002ool00`00ooooo`06
ool00`00ool0000500000ooo0000000200000ooo003oo`0500000ooo0000000200000ooo003oo`1N
ool01000ool00?oo100000?oo`0000000P000ooo100000?oo`0000000`0000?oo`000000100000?o
o`0000000P0000?oo`00ool0100000?oo`0000000`0000?oo`000000AOoo0009ool01@00oooooooo
00000ooo00<00?ooool01_oo00D00?ooooooo`0000;oo`03003ooooo00Koo`P00003ool000000080
0003ool00?oo00D00003ool0000000800003ool00?oo05koo`<00003ool0000000800003ool00000
008000;oo`03003oo`0000800003ool0000000<00003ool0000000@00003ool0000000800003ool0
0?oo00@00003ool0000000<00004ool000000014ool000Ooo`<000koo`<000?oo`04003ooooo0008
ool00`00ool0000500000ooo0000000200000ooo003oo`0500000ooo0000000200000ooo003oo`1M
ool400000ooo0000000200000ooo000000020002ool00`00ool0000200000ooo0000000300000ooo
0000000400000ooo0000000200000ooo003oo`0400000ooo0000000300001?oo00000000A?oo0009
ool00`00ooooo`0>ool00`00ooooo`02ool20009ool00`00ool0000500000ooo0000000200000ooo
003oo`0400000ooo0000000300000ooo003oo`1Mool400000ooo0000000300001?oo000000000_oo
00D00?oo00000?oo00H00003ool0000000@00003ool0000000800003ool00?oo00@00003ool00000
00<00003ool00000008004?oo`00:_oo00<00?oo00001@0000?oo`0000000P0000?oo`00ool01000
00?oo`0000000`0000?oo`00ool0GOoo100000?oo`0000000`0000Coo`00000000;oo`05003oo`00
003oo`0600000ooo0000000400000ooo0000000200000ooo003oo`0400000ooo0000000200000ooo
000000030013ool002Coo`<000?oo`03003oo`0000D00003ool0000000800003ool00?oo00@00003
ool0000000<00003ool00?oo05_oo`03003oo`0000<00003ool0000000<00006ool00000003oo`00
0_oo0P0000?oo`00ool0100000?oo`000000100000?oo`0000000P0000?oo`00ool0100000?oo`00
00000P0000?oo`0000000P0000?oo`00ool0@Ooo000Zool600000ooo003oo`0400000ooo003oo`04
00000ooo0000000300001Ooo003ooooo0000FOoo00<00?oo00000`0000?oo`0000000`0000Koo`00
00000?oo0002ool200000ooo003oo`0400000ooo0000000400000ooo0000000200000ooo003oo`04
00000ooo0000000200000ooo0000000200000ooo003oo`11ool002Woo`@00006ool00000ool00?oo
100000?oo`00ool0100000?oo`0000000`0000Goo`00003oo`0005Woo`03003oo`0000800003ool0
000000@00006ool00000003oo`000_oo0P0000?oo`00ool0100000?oo`000000100000?oo`000000
0P0000?oo`00ool0100000?oo`0000000P0000?oo`0000000P000_oo00<00?ooool0?ooo000Yool4
00001_oo00000?oo003oo`@00003ool00?oo00@00003ool0000000<00003ool00000008005Woo`03
003oo`0000800003ool0000000@00006ool00000003oo`000_oo0P0000?oo`00ool0100000?oo`00
0000100000?oo`0000000P0000?oo`00ool0100000?oo`0000000P0000?oo`0000000P000_oo00<0
0?ooool0?ooo000Zool01000ool00?oo0P0000?oo`00ool0100000?oo`00ool0100000?oo`000000
0`0000?oo`0000000P00FOoo00<00?oo00000P0000?oo`000000100000?oo`0000000P000ooo0P00
00?oo`00ool0100000?oo`000000100000?oo`0000000P0000?oo`00ool0100000?oo`0000000P00
00?oo`00ool00P000ooo00<00?ooool0?_oo000Zool01000ool00?oo0P0000?oo`00ool0100000?o
o`00ool0100000?oo`0000000`0000?oo`0000000P00FOoo1@0000?oo`000000100000Goo`00003o
o`0000?oo`800003ool00?oo00@00003ool0000000@00003ool0000000800003ool00?oo00@00003
ool0000000800003ool00?oo008000?oo`03003ooooo03koo`00:_oo00@00?oo003oo`800005ool0
0?oo003oo`0200000ooo003oo`0400000ooo0000000300000ooo00000002001Iool500000ooo0000
000400001Ooo00000?oo00000ooo0P0000?oo`00ool0100000?oo`000000100000?oo`0000000P00
00?oo`00ool0100000?oo`0000000P0000?oo`00ool00P000ooo00<00?ooool0?_oo000Zool30000
2?oo00000?oo003oo`00ool200000ooo003oo`0400000ooo0000000300000ooo00000002001Gool0
0`00ool0000400000ooo0000000400001Ooo00000?oo00000ooo0P0000?oo`00ool0100000?oo`00
ool0100000?oo`0000000P0000?oo`00ool0100000?oo`0000000P0000?oo`00ool00P000ooo00<0
0?ooool0?_oo000Zool01000ool00?oo0P0000Goo`00ool00?oo00800003ool00?oo00@00003ool0
000000<00003ool0000000800003ool00?oo05Coo`03003oo`0000@00003ool0000000@00005ool0
0000ool00003ool200000ooo003oo`0400000ooo003oo`0400000ooo0000000200000ooo003oo`04
00000ooo0000000200000ooo003oo`020004ool00`00ooooo`0mool002[oo`04003oo`00ool20000
1Ooo003oo`00ool00P0000?oo`00ool00P0000?oo`00ool01@0000?oo`0000000P0000?oo`00ool0
E?oo00<00?oo0000100000?oo`000000100000Goo`00003oo`0000?oo`800003ool00?oo00@00003
ool00?oo00@00003ool0000000800003ool00?oo00@00003ool0000000800003ool00?oo008000Co
o`03003ooooo03goo`00:_oo00@00?oo003oo`800005ool00?oo003oo`0200000ooo003oo`020000
0ooo003oo`0500000ooo0000000200000ooo003oo`1Dool00`00ool0000400000ooo000000020000
0ooo003oo`0200000ooo003oo`02ool200000ooo003oo`0400000ooo003oo`0400000ooo00000002
00000ooo003oo`0400000ooo0000000200000ooo003oo`020004ool00`00ooooo`0mool002[oo`04
003oo`00ool200001Ooo003oo`00ool00P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000
1@00EOoo00<00?oo0000100000?oo`0000000P0000?oo`00ool00P0000?oo`00ool00_oo0P0000?o
o`00ool0100000?oo`00ool00P0000?oo`00ool0100000?oo`00ool0100000?oo`0000000P0000?o
o`00ool00P001?oo00<00?ooool0?Ooo000Zool01000ool00?oo0P0000Goo`00ool00?oo00800003
ool00?oo00800003ool00?oo00@00003ool0000000D005Goo`03003oo`0000800003ool00?oo00@0
0003ool00?oo00800003ool00?oo00;oo`800003ool00?oo00@00003ool00?oo00800003ool00?oo
00@00003ool00?oo00@00003ool0000000800003ool00?oo008000Goo`03003ooooo03coo`00:_oo
0`0000Soo`00003oo`00ool00?oo0P0000?oo`00ool00P0000Goo`00ool00?oo00800003ool00?oo
00D005Coo`800007ool00000003oo`00ool0100000?oo`00ool00P0000?oo`00ool00_oo0P0000?o
o`00ool0100000?oo`00ool00P0000Goo`00ool00?oo00800003ool00?oo00@00003ool00?oo0080
0003ool00?oo008000Goo`03003ooooo03coo`00:_oo00@00?oo003oo`800005ool00?oo003oo`02
00000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool01@003?oo1000;_oo00<00?ooool0
4_oo0`0000Ooo`0000000?oo003oo`0400000ooo003oo`0200000ooo003oo`02ool400000ooo0000
000200000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool0100000?oo`00ool00P0000?o
o`00ool00P001Ooo00<00?ooool01?oo0P00:_oo10000ooo0P000ooo000Zool01000ool00?oo0P00
00Goo`00ool00?oo00800004ool00?oo0002ool01000ool00?oo0P0000?oo`00ool01@0000?oo`00
ool02Ooo00<00?ooool0;ooo00<00?ooool04Ooo100000Ooo`0000000?oo003oo`0400000ooo003o
o`0200000ooo003oo`02ool200000ooo0000000400000ooo003oo`0200001Ooo003oo`00ool00P00
00?oo`00ool0100000?oo`00ool00P0000?oo`00ool00P001Ooo00<00?ooool00ooo00@00?ooool0
02_oo`05003oooooool00002ool10001ool1ool002[oo`04003oo`00ool200001Ooo003oo`00ool0
0P0000Coo`00ool000;oo`04003oo`00ool200000ooo003oo`0500000ooo003oo`0:ool00`00oooo
o`0[ool5000Bool400002Ooo00000000ool00?oo003oo`0200000ooo003oo`0200000ooo003oo`02
ool200000ooo0000000400000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool0100000?o
o`00ool00P0000?oo`00ool00P001_oo00<00?ooool00_oo00@00?ooool002_oo`05003oooooool0
0002ool10001ool1ool002[oo`04003oo`00ool200001Ooo003oo`00ool00P0000Coo`00ool000;o
o`04003oo`00ool200000ooo003oo`0400001?oo000000002ooo00<00?ooool0:_oo00@00?ooool0
01?oo`<00003ool0000000800005ool00?oo003oo`0200000ooo003oo`0200000ooo003oo`02ool4
00000ooo0000000200000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool00P0000Goo`00
ool00?oo00800003ool00?oo008000Koo`03003ooooo00?oo`8002coo`05003oooooool00002ool1
0001ool1ool002[oo`<00008ool00000ool00?oo003oo`800004ool00?oo0002ool01000ool00?oo
0P0000?oo`00ool0100000Coo`00000000coo`03003ooooo02[oo`03003oo`0001?oo`<00003ool0
0?oo00800005ool00?oo003oo`0200000ooo003oo`0200000ooo003oo`03ool01000ool00?oo0P00
00Goo`00ool00?oo00800005ool00?oo003oo`0200000ooo003oo`0200001Ooo003oo`00ool00P00
00?oo`00ool00P001_oo00<00?ooool00_oo00@00?ooool002_oo`05003oooooool00002ool10001
ool1ool002[oo`04003oo`00ool200001Ooo003oo`00ool00P0000Coo`00ool000;oo`04003oo`00
ool200000ooo003oo`0400001?oo000000002Ooo00@00?ooool002goo`8001?oo`<00003ool00?oo
00800005ool00?oo003oo`0200000ooo003oo`0200000ooo003oo`03ool01000ool00?oo0P0000Go
o`00ool00?oo00800005ool00?oo003oo`0200000ooo003oo`0200001?oo003oo`000_oo00@00?oo
003oo`8000Ooo`05003oooooool00002ool00`00ooooo`0Wool30003ool01000ooooo`000_oo000Z
ool01@00ool00?oo00000_oo00@00?oo003oo`800004ool00?oo0002ool01000ool00?oo0P0000?o
o`00ool0100000Coo`00000000[oo`8002ooo`03003ooooo017oo`<00003ool00?oo00800005ool0
0?oo003oo`0200000ooo003oo`0200000ooo003oo`03ool50002ool01000ool00?oo0P0000Coo`00
ool000;oo`04003oo`00ool200001?oo003oo`000_oo00@00?oo003oo`8000Ooo`03003ooooo00;o
o`8002coo`03003ooooo00;oo`8000?oo`00:_oo00D00?oo003oo`0000;oo`05003oo`00ool00002
ool00`00ool00002ool00`00ool00002ool01000ool00?oo100000Coo`00000004goo`03003oo`00
00800003ool00?oo00800005ool00?oo003oo`0200001?oo003oo`000_oo00<00?ooool00_oo00D0
0?oo003oo`0000;oo`05003oooooool00002ool00`00ool00002ool01@00ool00?oo00000_oo00<0
0?oo00000_oo00@00?oo003oo`8000Soo`03003ooooo03Woo`00:_oo00D00?oo003oo`0000;oo`05
003oo`00ool00002ool00`00ool00002ool00`00ool00002ool01000ool00?oo100000?oo`00ool0
0P00C?oo1@0000?oo`00ool00P0000Coo`00ool000;oo`05003oo`00ool00002ool00`00ooooo`02
ool01@00ool00?oo00000_oo00D00?ooooooo`0000Coo`08003ooooo003oo`00ool000;oo`03003o
o`0000;oo`04003oo`00ool20008ool00`00ooooo`0iool002[oo`05003oo`00ool00002ool01@00
oooooooo00001ooo00<00?ooool00_oo00H00?oo003oo`00ool200000ooo003oo`03001;ool50000
1?oo003oo`000_oo00<00?ooool00_oo00D00?oo003oo`0000;oo`03003ooooo00;oo`05003oo`00
ool00002ool00`00ooooo`09ool01@00oooooooo00000_oo00<00?oo00000_oo00D00?ooooooo`00
00[oo`03003ooooo03Soo`009?oon`000Ooo000Zool00`00ooooo`09ool00`00ooooo`09ool00`00
ooooo`08ool01000ooooo`000ooo0P0000Coo`00ool000;oo`03003oo`0000;oo`05003oo`00ool0
000;ool00`00ooooo`02ool00`00ool00002ool00`00ool00004ool00`00ooooo`02ool01@00ool0
0?oo00000_oo00<00?oo00000_oo00@00?oo003oo`800003ool00?oo00Goo`03003ooooo00Woo`03
003ooooo00Woo`03003ooooo00Woo`03003ooooo00Woo`03003ooooo00Woo`03003ooooo00Goo`07
003oooooool00?oo00000_oo00D00?oo003oo`0000;oo`03003oo`0000Koo`04003ooooo0003ool0
1P00ooooo`00ool000;oo`03003ooooo00Koo`04003ooooo0006ool002[oo`03003ooooo02?oo`04
003oooooool200001?oo003oo`000_oo00<00?oo00000_oo00D00?oo003oo`0000Woo`03003oo`00
00Coo`03003oo`0000;oo`03003oo`0000Coo`03003oo`0000;oo`05003oo`00ool00002ool00`00
ool00002ool01000ool00?oo0`007ooo00<00?ooool0:_oo00H00?ooool00?oo0002ool01@00ool0
0?oo00001?oo00<00?ooool01?oo00P00?ooool00?oo003oo`000_oo00<00?oo00000_oo00<00?oo
00001_oo00@00?ooool000Koo`00:_oo00<00?ooool08ooo00@00?ooooooo`800004ool00?oo0002
ool00`00ool00002ool01@00ool00?oo00001?oo00P00?ooool00?oo003oo`000_oo00D00?oo003o
o`0000;oo`03003oo`0000;oo`05003oo`00ool00002ool01@00ool00?oo00000_oo00<00?oo0000
0_oo00<00?oo00001000C?oo00<00?ooool00_oo00P00?oo00000?oo003oo`000_oo00@00?oo003o
o`800004ool00?oo0002ool01@00ool00?oo00000_oo00<00?oo00000_oo00D00?oo003oo`0000Co
o`03003oo`0000Ooo`00:_oo00<00?ooool08ooo00@00?ooooooo`800004ool00?oo0002ool01000
ool00?oo0P0000Coo`00ool000;oo`03003oo`0000;oo`05003oo`00ool00002ool01@00ool00?oo
00000_oo00<00?oo00000_oo00@00?oo003oo`800005ool00?oo003oo`0200001?oo003oo`000_oo
00<00?oo00001000COoo00H00?ooooooo`00ool200001?oo003oo`000_oo00@00?oo003oo`800005
ool00?oo003oo`0200001?oo003oo`000_oo00<00?oo00000_oo00D00?oo003oo`0000;oo`05003o
o`00ool00007ool002[oo`03003ooooo02Coo`03003ooooo00800004ool00?oo0002ool01000ool0
0?oo0P0000Coo`00ool000;oo`04003oo`00ool200001?oo003oo`000_oo00D00?oo003oo`0000;o
o`04003oo`00ool200000ooo003oo`0200001Ooo003oo`00ool00P0000Coo`00ool000;oo`03003o
o`0000800003ool00?oo04coo`06003oooooool00?oo0P0000Coo`00ool000;oo`04003oo`00ool2
00001Ooo003oo`00ool00P0000Coo`00ool000;oo`03003oo`0000;oo`05003oo`00ool00002ool0
1@00ool00?oo00001ooo000Zool2000Uool00`00ooooo`0200000ooo003oo`0200001Ooo003oo`00
ool00P0000Coo`00ool000;oo`04003oo`00ool200001?oo003oo`000_oo00D00?oo003oo`0000;o
o`04003oo`00ool200000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool00P0000?oo`00
ool00`00D?oo00D00?ooool00?oo00@00008ool00000ool00?oo003oo`800005ool00?oo003oo`02
00001?oo003oo`000_oo00@00?oo003oo`800004ool00?oo0002ool01@00ool00?oo00001ooo000Z
ool00`00ooooo`0Uool01`00ool00000ool00?oo00800005ool00?oo003oo`0200001?oo003oo`00
0_oo00@00?oo003oo`800004ool00?oo0002ool01@00ool00?oo00000_oo00@00?oo003oo`800003
ool00?oo00800005ool00?oo003oo`0200000ooo003oo`0200000ooo003oo`03001@ool01@00ool0
0000ool0100000Soo`00003oo`00ool00?oo0P0000Goo`00ool00?oo00800004ool00?oo0002ool0
1000ool00?oo0P0000Coo`00ool000;oo`05003oo`00ool00007ool002[oo`03003ooooo02Goo`07
003oo`00003oo`00ool00P0000?oo`00ool0100000?oo`00ool00P0000Goo`00ool00?oo00800004
ool00?oo0002ool01@00ool00?oo00000_oo00<00?oo00000`0000?oo`00ool00P0000?oo`00ool0
100000?oo`00ool0100000Coo`000000053oo`04003oo`00ool500002?oo00000?oo003oo`00ool2
00000ooo003oo`0400001Ooo003oo`00ool00P0000?oo`00ool00P0000Coo`00ool000;oo`05003o
o`00ool00007ool002[oo`03003ooooo02Koo`<00003ool0000000@00003ool0000000800003ool0
0?oo00800005ool00?oo003oo`0200001?oo003oo`000_oo00@00?oo003oo`800003ool00?oo00@0
0003ool00?oo00800003ool00?oo00@00003ool00?oo00@00004ool00000001Aool200000ooo0000
000300002?oo00000?oo003oo`00ool200000ooo003oo`0400001Ooo003oo`00ool00P0000?oo`00
ool00P0000Coo`00ool000;oo`05003oo`00ool00007ool002[oo`03003ooooo02Koo`<00003ool0
000000@00003ool0000000800003ool00?oo00800005ool00?oo003oo`0200001?oo003oo`000_oo
00@00?oo003oo`800003ool00?oo00@00003ool00?oo00800003ool00?oo00@00003ool00?oo00@0
0004ool00000001Aool200000ooo0000000300002?oo00000?oo003oo`00ool200000ooo003oo`04
00001Ooo003oo`00ool00P0000?oo`00ool00P0000Coo`00ool000;oo`05003oo`00ool00007ool0
02[oo`8002Ooo`@00003ool0000000<00003ool0000000800003ool00?oo00800005ool00?oo003o
o`0200000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool0100000?oo`00ool0100000?o
o`0000000P0000?oo`00ool0100000?oo`000000D_oo0P0000?oo`0000000`0000?oo`0000000P00
00?oo`00ool00P0000?oo`00ool0100000?oo`00ool0100000?oo`00ool00P0000Coo`00ool000;o
o`05003oo`00ool00007ool002[oo`03003ooooo02Koo`@00003ool0000000<00003ool000000080
0003ool00?oo00800005ool00?oo003oo`0200000ooo003oo`0200001Ooo003oo`00ool00P0000?o
o`00ool0100000?oo`00ool0100000?oo`0000000P0000?oo`00ool0100000?oo`000000D_oo0P00
00?oo`0000000`0000?oo`0000000P0000?oo`00ool00P0000?oo`00ool0100000?oo`00ool01000
00?oo`00ool00P0000Coo`00ool000;oo`05003oo`00ool00007ool002[oo`03003ooooo02Koo`@0
0003ool0000000<00003ool0000000800003ool00?oo00800005ool00?oo003oo`0200000ooo003o
o`0200001Ooo003oo`00ool00P0000?oo`00ool0100000?oo`00ool0100000?oo`0000000P0000?o
o`00ool01@00EOoo00<00?oo0000100000?oo`0000000P0000?oo`00ool0100000?oo`0000000P00
00?oo`00ool0100000?oo`00ool00P0000Coo`00ool000;oo`05003oo`00ool00007ool002[oo`03
003ooooo02Soo`800003ool0000000<00003ool0000000800003ool00?oo00800005ool00?oo003o
o`0400002?oo00000?oo003oo`00ool200000ooo003oo`0400000ooo003oo`0400000ooo00000002
00000ooo003oo`05001Eool00`00ool0000400000ooo0000000200000ooo003oo`0400000ooo0000
000200000ooo003oo`0400000ooo003oo`0200001?oo003oo`000_oo00D00?oo003oo`0000Ooo`00
:_oo00<00?ooool0:?oo0P0000?oo`0000000`0000?oo`0000000P0000?oo`00ool00P0000Goo`00
ool00?oo00@00008ool00000ool00?oo003oo`800003ool00?oo00@00003ool00?oo00@00003ool0
000000<00003ool0000000@005Ooo`D00003ool0000000800003ool0000000@00003ool000000080
0003ool00?oo00@00003ool00?oo00800004ool00?oo0002ool01@00ool00?oo00001ooo000Zool0
0`00ooooo`0Xool200000ooo0000000300000ooo0000000200000ooo003oo`0400000ooo003oo`04
00000ooo0000000200000ooo003oo`0200000ooo003oo`0400000ooo003oo`0400000ooo00000003
00000ooo0000000200000ooo003oo`1Fool500000ooo0000000200000ooo0000000400000ooo0000
000200000ooo003oo`0400000ooo003oo`0200000ooo003oo`0200001_oo003oo`00ool000Ooo`00
:_oo0P00:Ooo0P0000?oo`0000000`0000?oo`0000000P0000?oo`00ool0100000?oo`0000001000
00?oo`0000000P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000100000?oo`0000000`00
00?oo`0000000P0000?oo`00ool0E_oo1@0000?oo`0000000P0000?oo`000000100000?oo`000000
0P0000?oo`00ool0100000?oo`00ool00P0000?oo`00ool0100000Coo`00ool000Ooo`00:_oo00<0
0?ooool0:Ooo00<00?oo0000100000?oo`0000000P0000?oo`00ool0100000?oo`000000100000?o
o`0000000P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000100000?oo`0000000`0000?o
o`0000000P0000?oo`00ool0E_oo1@0000?oo`0000000P0000?oo`000000100000?oo`0000000P00
00?oo`00ool0100000?oo`00ool00P0000?oo`00ool0100000Coo`00ool000Ooo`00:_oo00<00?oo
ool0:Ooo00<00?oo0000100000?oo`0000000P0000?oo`000000100000?oo`000000100000?oo`00
00000P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000100000?oo`0000000`0000?oo`00
00000P0000?oo`00ool0E_oo1@0000?oo`0000000P0000?oo`000000100000?oo`0000000P0000?o
o`00ool0100000?oo`00ool00P0000?oo`00ool0100000Coo`00ool000Ooo`00:_oo00<00?ooool0
:Ooo00<00?oo0000100000?oo`0000000P0000?oo`000000100000?oo`000000100000?oo`000000
0P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000100000?oo`0000000`0000?oo`000000
0P00FOoo1@0000?oo`0000000P0000?oo`000000100000?oo`0000000P0000?oo`00ool0100000?o
o`00ool00P0000?oo`00ool0100000Coo`00ool000Ooo`00:_oo00<00?ooool0:ooo1@0000?oo`00
00000P0000?oo`000000100000?oo`000000100000?oo`0000000P0000?oo`00ool00P0000?oo`00
ool0100000?oo`000000100000?oo`0000000`0000?oo`0000000P00FOoo1@0000?oo`0000000P00
00?oo`000000100000?oo`0000000P0000?oo`00ool0100000?oo`00ool00P0000?oo`00ool01000
00Coo`00ool000Ooo`00:_oo00<00?ooool0:ooo1@0000?oo`0000000`0000?oo`0000000`0000?o
o`000000100000?oo`0000000P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000100000?o
o`0000000`0000?oo`0000000P00FOoo00<00?oo00000P0000?oo`0000000P0000?oo`0000001000
00?oo`0000000P0000?oo`00ool0100000?oo`00ool00P0000?oo`00ool0100000Coo`00ool000Oo
o`00:_oo0P00;?oo1@0000?oo`0000000`0000?oo`0000000`0000?oo`000000100000?oo`000000
0P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000100000?oo`0000000`0000?oo`000000
0P00Fooo0P0000?oo`000000100000?oo`0000000`0000?oo`0000000P0000?oo`00ool0100000?o
o`00ool00P0000?oo`00ool0100000Coo`00ool000Ooo`00:_oo00<00?ooool0:ooo1@0000?oo`00
00000`0000?oo`0000000`0000?oo`000000100000?oo`0000000P0000?oo`00ool00P0000?oo`00
ool0100000?oo`000000100000?oo`0000000`0000?oo`0000000P00G?oo00<00?oo00001@0000?o
o`0000000`0000?oo`0000000P0000?oo`00ool0100000?oo`00ool00P0000?oo`00ool0100000Co
o`00ool000Ooo`001ooo10000_oo00D00?ooooooo`0000Koo`@000?oo`8000Woo`03003ooooo02_o
o`D00003ool0000000<00003ool0000000<00003ool0000000@00003ool0000000800003ool00?oo
00800003ool00?oo00@00003ool0000000@00003ool0000000<00003ool00000008005coo`03003o
o`0000D00003ool0000000<00003ool0000000800003ool00?oo00@00003ool00?oo00800003ool0
0?oo00@00004ool00?oo0007ool000Woo`03003ooooo00;oo`03003oo`0000Woo`05003oooooool0
0002ool00`00ooooo`06ool00`00ooooo`0[ool00`00ool0000300000ooo0000000200000ooo0000
000300000ooo0000000400000ooo0000000200000ooo003oo`0200000ooo003oo`0400000ooo0000
000400000ooo0000000300001Ooo00000?oo0000G?oo00<00?oo00001@0000?oo`0000000`0000?o
o`0000000P0000?oo`00ool0100000?oo`00ool00P0000?oo`00ool0100000Coo`00ool000Ooo`00
2Ooo00<00?ooool00ooo00<00?ooool02?oo00D00?ooooooo`0000;oo`03003ooooo00Koo`03003o
oooo02_oo`03003oo`0000<00003ool0000000800003ool0000000<00003ool0000000@00003ool0
000000800003ool00?oo00800003ool00?oo00@00003ool0000000@00003ool0000000<00003ool0
000005koo`04003oo`00ool400000ooo0000000300000ooo0000000200000ooo003oo`0400000ooo
003oo`0200000ooo003oo`0400001?oo003oo`001ooo0009ool00`00ooooo`02ool00`00ool00009
ool01@00oooooooo00000_oo00<00?ooool01_oo0P00;_oo100000?oo`0000000P0000?oo`000000
0`0000?oo`000000100000?oo`0000000P0000?oo`00ool00P0000?oo`00ool0100000?oo`000000
100000?oo`0000000`0000?oo`000000G_oo00@00?oo003oo`@00003ool0000000<00003ool00000
00800003ool00?oo00@00003ool00?oo00800003ool00?oo00D00003ool0000000Ooo`002Ooo00D0
0?ooooooo`0000?oo`03003ooooo00Koo`05003oooooool00002ool00`00ooooo`06ool00`00oooo
o`0]ool400000ooo0000000200000ooo0000000300000ooo0000000400000ooo0000000200000ooo
0000000400000ooo0000000200000ooo0000000400000ooo0000000300000ooo0000001Pool00`00
ool0000300000ooo0000000300000ooo0000000200000ooo003oo`0400000ooo0000000200000ooo
003oo`0500000ooo00000007ool000Ooo`<000koo`<000?oo`04003ooooo0008ool00`00ooooo`0]
ool400000ooo0000000200000ooo0000000300000ooo0000000400000ooo0000000200000ooo0000
000400000ooo0000000200000ooo0000000400000ooo0000000300000ooo003oo`1Pool00`00ool0
000300000ooo0000000300000ooo0000000200000ooo003oo`0400000ooo0000000200000ooo003o
o`0500000ooo00000007ool000Woo`03003ooooo00koo`03003ooooo00;oo`8000Woo`03003ooooo
02goo`@00003ool0000000800003ool0000000<00003ool0000000@00003ool0000000800003ool0
000000@00003ool0000000800003ool0000000@00003ool0000000<00003ool00?oo063oo`03003o
o`0000<00003ool0000000<00003ool0000000800003ool00?oo00@00003ool0000000800003ool0
0?oo00D00003ool0000000Ooo`009?oo0`000ooo00<00?ooool0;Ooo100000?oo`0000000P0000?o
o`0000000`0000?oo`000000100000?oo`0000000`0000?oo`0000000`0000?oo`0000000`0000?o
o`0000000`0000?oo`0000000`0000?oo`00ool0H?oo00<00?oo00000`0000?oo`0000000`0000?o
o`0000000P0000?oo`00ool0100000?oo`0000000P0000?oo`00ool01@0000?oo`0000001ooo000Z
ool2000^ool400000ooo0000000200000ooo0000000300000ooo0000000400000ooo000000030000
0ooo0000000300000ooo0000000300000ooo0000000300000ooo0000000300000ooo003oo`1Pool0
0`00ool0000300000ooo0000000300000ooo0000000200000ooo003oo`0400000ooo000000020000
0ooo003oo`0500000ooo00000007ool002Woo`<002ooo`04003oo`00ool400000ooo000000030000
0ooo0000000400000ooo0000000300000ooo0000000300000ooo0000000300000ooo000000030000
0ooo00000003001Sool00`00ool0000300000ooo0000000300000ooo0000000200000ooo003oo`04
00000ooo0000000200000ooo003oo`0500000ooo00000007ool002Woo`<0037oo`03003oo`0000<0
0003ool0000000@00003ool0000000<00003ool0000000<00003ool0000000<00003ool0000000<0
0003ool0000000800003ool0000000@006?oo`03003oo`0000<00003ool0000000<00003ool00000
00800003ool0000000@00003ool0000000800003ool00?oo00D00003ool0000000Ooo`00:_oo00<0
0?ooool0<?oo00<00?oo00000`0000?oo`000000100000?oo`0000000`0000?oo`0000000`0000?o
o`0000000`0000?oo`0000000`0000?oo`0000000P0000?oo`0000001000IOoo100000?oo`000000
0`0000?oo`0000000P0000?oo`000000100000?oo`0000000P0000?oo`00ool01@0000?oo`000000
1ooo000Zool00`00ooooo`0`ool00`00ool0000300000ooo0000000400000ooo0000000300000ooo
0000000300000ooo0000000300000ooo0000000300000ooo0000000200000ooo003oo`0200000ooo
003oo`1Tool400000ooo0000000300000ooo0000000200000ooo0000000400000ooo000000040000
0ooo0000000300000ooo00000007ool002[oo`03003ooooo033oo`03003oo`0000<00003ool00000
00800003ool00?oo00D00003ool0000000<00003ool0000000<00003ool0000000<00003ool00000
00800003ool00?oo00800003ool00?oo06Coo`@00003ool0000000<00003ool0000000800003ool0
000000@00003ool0000000@00003ool0000000<00003ool0000000Ooo`00:_oo00<00?ooool0<?oo
00<00?oo00000`0000?oo`0000000P0000?oo`00ool01@0000?oo`0000000`0000?oo`0000000`00
00?oo`0000000`0000?oo`0000000P0000?oo`00ool00P0000?oo`00ool0I?oo100000?oo`000000
0P0000?oo`0000000`0000?oo`000000100000?oo`000000100000?oo`0000000`0000?oo`000000
1ooo000Zool2000aool00`00ool0000300000ooo0000000200000ooo003oo`0500000ooo00000003
00000ooo0000000300000ooo0000000300000ooo0000000200000ooo003oo`02001Wool400000ooo
0000000200000ooo0000000300000ooo0000000400000ooo0000000400000ooo0000000300000ooo
00000007ool002[oo`03003ooooo03;oo`@00003ool0000000800003ool00?oo00D00003ool00000
00<00003ool0000000<00003ool0000000<00003ool0000000800003ool00?oo008006Soo`<00003
ool0000000800003ool0000000<00003ool0000000@00003ool0000000@00003ool0000000<00003
ool0000000Ooo`00:_oo00<00?ooool0<_oo100000?oo`0000000P0000?oo`00ool01@0000?oo`00
00000`0000?oo`0000000`0000?oo`0000000`0000?oo`0000000P0000?oo`00ool00P00J_oo00<0
0?oo00000`0000?oo`0000000`0000?oo`000000100000?oo`000000100000Ooo`0000000?oo003o
o`020007ool002[oo`03003ooooo03;oo`07003oo`00003oo`00ool00P0000?oo`00ool01@0000?o
o`0000000`0000?oo`0000000`0000?oo`0000000`0000?oo`0000000P0000?oo`00ool00P00J_oo
00<00?oo00000`0000?oo`000000100000?oo`0000000`0000?oo`000000100000Ooo`0000000?oo
003oo`020007ool002[oo`03003ooooo03;oo`07003oo`00003oo`00ool00P0000?oo`00ool01@00
00?oo`0000000`0000?oo`0000000`0000?oo`0000000`0000?oo`0000000P0000?oo`00ool00P00
J_oo00<00?oo00000`0000?oo`000000100000?oo`0000000`0000?oo`0000000`0000?oo`000000
0P0000?oo`00ool00P001ooo000Zool00`00ooooo`0bool01`00ool00000ool00?oo00800003ool0
0?oo00D00003ool0000000<00003ool0000000<00003ool0000000<00003ool0000000800003ool0
0?oo06coo`04003oo`00ool200000ooo0000000400000ooo0000000300000ooo0000000300000ooo
0000000200000ooo003oo`020007ool002[oo`8003?oo`07003oo`00003oo`00ool00P0000?oo`00
ool01@0000?oo`0000000`0000?oo`0000000`0000?oo`0000000`0000?oo`0000000P0000?oo`00
ool0K?oo00@00?oo003oo`800003ool0000000@00003ool0000000<00003ool0000000<00003ool0
000000800003ool00?oo008000Ooo`00:_oo00<00?ooool0=?oo0P0000?oo`00ool00P0000?oo`00
ool01@0000?oo`0000000`0000?oo`0000000`0000?oo`0000000`0000?oo`0000000P0000?oo`00
ool0K?oo00@00?oo003oo`800003ool0000000@00003ool0000000<00003ool0000000<00003ool0
000000800003ool00?oo008000Ooo`00:_oo00<00?ooool0=?oo0P0000?oo`00ool00P0000?oo`00
ool01@0000?oo`0000000`0000?oo`0000000P0000?oo`000000100000?oo`0000000P0000?oo`00
ool0K_oo00L00?oo00000?oo003oo`0400000ooo0000000300000ooo0000000300000ooo00000002
00000ooo003oo`020007ool002[oo`03003ooooo03Coo`800003ool00?oo00800003ool00?oo00@0
0003ool0000000@00003ool0000000800003ool0000000@00003ool0000000800003ool00?oo06ko
o`07003oo`00003oo`00ool0100000?oo`0000000`0000?oo`0000000`0000?oo`0000000P0000?o
o`00ool00P001ooo000Zool00`00ooooo`0eool01000ool00?oo0P0000?oo`00ool0100000?oo`00
0000100000?oo`0000000P0000?oo`000000100000?oo`0000000P0000?oo`00ool0K_oo00L00?oo
00000?oo003oo`0400000ooo0000000300000ooo0000000300000ooo0000000200000ooo003oo`02
0007ool002[oo`8003Koo`04003oo`00ool200000ooo003oo`0400000ooo0000000400000ooo0000
000200000ooo0000000400000ooo003oo`0200000ooo003oo`1^ool01`00ool00000ool00?oo00@0
0003ool0000000<00003ool0000000<00003ool0000000800003ool00?oo008000Ooo`00:_oo00<0
0?ooool0=Ooo00@00?oo003oo`800003ool00?oo00@00003ool0000000@00003ool0000000800003
ool0000000@00003ool00?oo0080077oo`07003oo`00003oo`00ool0100000?oo`0000000P0000?o
o`000000100000?oo`0000000P0000?oo`00ool00P001ooo000Zool00`00ooooo`0gool01`00ool0
0000ool00?oo00@00003ool00?oo00@00003ool0000000800003ool0000000@00003ool00?oo0080
07?oo`800003ool00?oo00@00003ool0000000800003ool0000000@00003ool0000000800003ool0
0?oo008000Ooo`001ooo10000_oo00D00?ooooooo`0000Koo`@000?oo`8000Woo`03003ooooo03Oo
o`07003oo`00003oo`00ool0100000?oo`00ool0100000?oo`0000000P0000?oo`000000100000?o
o`00ool00P00Looo0P0000?oo`00ool0100000?oo`0000000P0000?oo`000000100000?oo`000000
0P0000?oo`00ool00P001ooo0007ool00`00ooooo`04ool00`00ool00009ool01@00oooooooo0000
0_oo00<00?ooool01_oo00<00?ooool0=ooo00L00?oo00000?oo003oo`0400000ooo003oo`040000
0ooo0000000200000ooo003oo`0400001?oo003oo`00M?oo0P0000?oo`00ool0100000?oo`000000
0P0000?oo`000000100000?oo`0000000P0000?oo`00ool00P001ooo0008ool00`00ooooo`04ool0
0`00ooooo`08ool01@00oooooooo00000_oo00<00?ooool01_oo00<00?ooool0=ooo00L00?oo0000
0?oo003oo`0400000ooo003oo`0400000ooo0000000200000ooo003oo`0400000ooo003oo`1eool2
00000ooo003oo`0400000ooo0000000200000ooo0000000400000ooo0000000200000ooo003oo`02
0007ool000Woo`03003ooooo00;oo`03003oo`0000Woo`05003oooooool00002ool00`00ooooo`06
ool2000hool01`00ool00000ool00?oo00@00003ool00?oo00@00003ool0000000800003ool00?oo
00800004ool00?oo001gool01000ool00?oo0P0000?oo`00ool0100000?oo`000000100000?oo`00
00000P0000Coo`00ool000Soo`002_oo00@00?ooool000?oo`03003ooooo00Koo`05003oooooool0
0002ool00`00ooooo`06ool00`00ooooo`0iool200000ooo003oo`0400000ooo003oo`0400000ooo
0000000200000ooo003oo`0200001?oo003oo`00NOoo00L00?oo00000?oo003oo`0400000ooo003o
o`0400000ooo0000000200001?oo003oo`002?oo0007ool01000ooooo`003Ooo0`000ooo00@00?oo
ool000Soo`03003ooooo03Woo`800005ool00?oo003oo`0200000ooo003oo`0400000ooo00000002
00000ooo003oo`0200001?oo003oo`00NOoo00L00?oo00000?oo003oo`0400000ooo003oo`040000
0ooo0000000200001?oo003oo`002?oo0008ool2000@ool00`00ooooo`02ool20009ool00`00oooo
o`0jool01@00ool00?oo00000_oo00@00?oo003oo`@00003ool0000000800003ool00?oo00800004
ool00?oo001iool01`00ool00000ool00?oo00@00003ool00?oo00@00003ool0000000800004ool0
0?oo0008ool002Coo`<000?oo`03003ooooo03[oo`05003oo`00ool00002ool01000ool00?oo1000
00?oo`0000000P0000?oo`00ool00P0000?oo`00ool0N_oo00L00?oo00000?oo003oo`0400000ooo
003oo`0400000ooo003oo`0200001?oo003oo`002?oo000Zool2000mool00`00ool00002ool01000
ool00?oo100000?oo`0000000P0000Coo`00ool000;oo`03003ooooo07Woo`07003oo`00003oo`00
ool0100000?oo`00ool0100000?oo`00ool00P0000Coo`00ool000Soo`00:Ooo0`00?Ooo00<00?oo
00000_oo00@00?oo003oo`@00003ool0000000800004ool00?oo0002ool00`00ooooo`1kool20000
0ooo003oo`0400000ooo003oo`0200001Ooo003oo`00ool00P0000Coo`00ool000Soo`00:Ooo0`00
?Ooo00<00?oo00000_oo00@00?oo003oo`@00003ool0000000800004ool00?oo0021ool01000ool0
0?oo100000?oo`00ool00P0000Goo`00ool00?oo00800003ool00?oo00Woo`00:_oo00<00?ooool0
?_oo00L00?ooool00?oo003oo`0400000ooo0000000200001?oo003oo`00POoo00H00?oo003oo`00
ool200000ooo003oo`0200001Ooo003oo`00ool00P0000?oo`00ool02Ooo000Zool00`00ooooo`0n
ool01`00ooooo`00ool00?oo00@00003ool0000000800004ool00?oo0023ool01000ool00?oo0P00
00?oo`00ool00P0000Goo`00ool00?oo00800003ool00?oo00Woo`00:_oo00<00?ooool0@Ooo00@0
0?oo003oo`@00003ool0000000800004ool00?oo0023ool01000ool00?oo0P0000?oo`00ool00P00
00Goo`00ool00?oo00800003ool00?oo00Woo`00:_oo0P00@_oo00@00?oo003oo`800005ool00?oo
003oo`0200001?oo003oo`00Pooo00@00?oo003oo`800004ool00?oo0002ool01000ool00?oo0P00
3?oo000Zool00`00ooooo`11ool01000ool00?oo0P0000Coo`00ool000;oo`03003oo`0008Ooo`08
003oo`00003oo`00ool000;oo`03003oo`0000ooo`00:_oo00<00?ooool0@ooo00P00?oo00000?oo
003oo`000_oo00<00?oo0000Qooo00P00?ooool00?oo003oo`000_oo00<00?oo00003ooo000Zool0
0`00ooooo`13ool02000ool00000ool00?oo0002ool00`00ooooo`27ool02000ooooo`00ool00?oo
0002ool00`00ool0000?ool002[oo`03003ooooo04?oo`08003ooooo003oo`00ool000;oo`03003o
oooo08[oo`05003oo`00ool0000Dool002[oo`03003ooooo04?oo`03003ooooo00;oo`03003oo`00
08ooo`05003oo`00ool0000Dool002[oo`8004Woo`03003ooooo097oo`03003oo`0001Coo`00:_oo
00<00?ooool0looo000Zool00`00ooooo`3cool002[oo`03003ooooo0??oo`00:_oo00<00?ooool0
looo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool00?ooob7oo`00oooo8Ooo003ooolQool0
0?ooob7oo`00oooo8Ooo003ooolQool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-1.76403, -3.11449*^-7, \
0.041865, 3.51908*^-9}}],

Cell[BoxData[
    TagBox[\(\[SkeletonIndicator]  Graphics  \[SkeletonIndicator]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[7]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simplification", "Subsection"],

Cell[TextData[{
  "The method ",
  StyleBox["LocallyExact", "MR"],
  " has an option ",
  StyleBox["SimplificationFunction", "MR"],
  " that can be used to simplify the results of ",
  StyleBox["DSolve", "MR"],
  "."
}], "Text"],

Cell["\<\
Here is the linearized component of the differential system that \
turns up in the splitting of the Lorenz equations using standard values for \
the parameters.\
\>", "Text"],

Cell[BoxData[{
    \(\(eqs = {\(Y\_1'\)[
                T] \[Equal] \[Sigma]\ \((Y\_2[T] - Y\_1[T])\), \(Y\_2'\)[
                T] \[Equal] 
              r\ Y\_1[T] - Y\_2[T], \(Y\_3'\)[
                T] \[Equal] \(-b\)\ Y\_3[T]} /. {\[Sigma] \[Rule] 10, 
            r \[Rule] 28, b \[Rule] 8/3};\)\), "\n", 
    \(\(ics\  = \ {Y\_1[0] \[Equal] \(-8\), Y\_2[0] \[Equal] 8, 
          Y\_3[0] \[Equal] 27};\)\), "\[IndentingNewLine]", 
    \(\(vars = {Y\_1[T], Y\_2[T], Y\_3[T]};\)\)}], "Input",
  CellLabel->"In[8]:="],

Cell[TextData[{
  "This subsystem is exactly solvable by ",
  StyleBox["DSolve", "MR"],
  ". "
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[eqs, vars, T]\)], "Input",
  CellLabel->"In[11]:="],

Cell[BoxData[
    \({{Y\_1[
            T] \[Rule] \(\(1\/2402\)\((\((1201\ \[ExponentialE]\^\(1\/2\ \
\((\(-11\) - \@1201)\)\ T\) + 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) + 1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\) - 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \
\@1201)\)\ T\))\)\ C[
                  1])\)\) - \(10\ \((\[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) - \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\))\)\ \
C[2]\)\/\@1201, 
        Y\_2[T] \[Rule] \(-\(\(28\ \((\[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) - \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\))\)\ C[
                      1]\)\/\@1201\)\) + \(\(1\/2402\)\((\((1201\ \
\[ExponentialE]\^\(1\/2\ \((\(-11\) - \@1201)\)\ T\) - 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) + 1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\) + 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \
\@1201)\)\ T\))\)\ C[2])\)\), 
        Y\_3[T] \[Rule] \[ExponentialE]\^\(\(\(-8\)\ T\)/3\)\ C[
              3]}}\)], "Output",
  CellLabel->"Out[11]="]
}, Open  ]],

Cell[TextData[{
  "Often the results of ",
  StyleBox["DSolve", "MR"],
  " can be simplified. This defines a function to simplify an expression and \
also prints out the input and the result."
}], "Text"],

Cell[BoxData[
    \(\(myfun[
          x_] := \[IndentingNewLine]Module[{simpx}, \
\[IndentingNewLine]Print["\<Before simplification \>", 
            x]; \[IndentingNewLine]simpx\  = \ 
            FullSimplify[
              ExpToTrig[
                x]]; \[IndentingNewLine]Print["\<After simplification \>", \ 
            simpx]; \[IndentingNewLine]simpx\[IndentingNewLine]];\)\)], \
"Input",
  CellLabel->"In[12]:="],

Cell[TextData[{
  "The function can be passed as an option to the method ",
  StyleBox["LocallyExact", "MR"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve[{eqs, \ ics}, \ vars, \ {T, \ 0, \ 1}, \ 
      StartingStepSize \[Rule] 1/10, \ 
      Method \[Rule] {"\<LocallyExact\>", \ "\<SimplificationFunction\>" \
\[Rule] myfun}]\)], "Input",
  CellLabel->"In[13]:="],

Cell[BoxData[
    InterpretationBox[\("Before simplification "\[InvisibleSpace]{\(\(1\/2402\
\)\((\((1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) - \@1201)\)\ T\) + 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) + 1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\) - 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \
\@1201)\)\ T\))\)\ Y\_1[
                  T])\)\) - \(10\ \((\[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) - \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\))\)\ \
Y\_2[T]\)\/\@1201, \(-\(\(28\ \((\[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) - \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\))\)\ \
Y\_1[T]\)\/\@1201\)\) + \(\(1\/2402\)\((\((1201\ \[ExponentialE]\^\(1\/2\ \((\
\(-11\) - \@1201)\)\ T\) - 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) - \
\@1201)\)\ T\) + 1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \@1201)\)\ T\) + 
                    9\ \@1201\ \[ExponentialE]\^\(1\/2\ \((\(-11\) + \
\@1201)\)\ T\))\)\ Y\_2[T])\)\), \[ExponentialE]\^\(\(\(-8\)\ T\)/3\)\ Y\_3[
              T]}\),
      SequenceForm[ "Before simplification ", {
        Plus[ 
          Times[ 
            Rational[ 1, 2402], 
            Plus[ 
              Times[ 1201, 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Times[ -1, 
                        Power[ 1201, 
                          Rational[ 1, 2]]]], T]]], 
              Times[ 9, 
                Power[ 1201, 
                  Rational[ 1, 2]], 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Times[ -1, 
                        Power[ 1201, 
                          Rational[ 1, 2]]]], T]]], 
              Times[ 1201, 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Power[ 1201, 
                        Rational[ 1, 2]]], T]]], 
              Times[ -9, 
                Power[ 1201, 
                  Rational[ 1, 2]], 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Power[ 1201, 
                        Rational[ 1, 2]]], T]]]], 
            Subscript[ Y, 1][ T]], 
          Times[ -10, 
            Power[ 1201, 
              Rational[ -1, 2]], 
            Plus[ 
              Power[ E, 
                Times[ 
                  Rational[ 1, 2], 
                  Plus[ -11, 
                    Times[ -1, 
                      Power[ 1201, 
                        Rational[ 1, 2]]]], T]], 
              Times[ -1, 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Power[ 1201, 
                        Rational[ 1, 2]]], T]]]], 
            Subscript[ Y, 2][ T]]], 
        Plus[ 
          Times[ -28, 
            Power[ 1201, 
              Rational[ -1, 2]], 
            Plus[ 
              Power[ E, 
                Times[ 
                  Rational[ 1, 2], 
                  Plus[ -11, 
                    Times[ -1, 
                      Power[ 1201, 
                        Rational[ 1, 2]]]], T]], 
              Times[ -1, 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Power[ 1201, 
                        Rational[ 1, 2]]], T]]]], 
            Subscript[ Y, 1][ T]], 
          Times[ 
            Rational[ 1, 2402], 
            Plus[ 
              Times[ 1201, 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Times[ -1, 
                        Power[ 1201, 
                          Rational[ 1, 2]]]], T]]], 
              Times[ -9, 
                Power[ 1201, 
                  Rational[ 1, 2]], 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Times[ -1, 
                        Power[ 1201, 
                          Rational[ 1, 2]]]], T]]], 
              Times[ 1201, 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Power[ 1201, 
                        Rational[ 1, 2]]], T]]], 
              Times[ 9, 
                Power[ 1201, 
                  Rational[ 1, 2]], 
                Power[ E, 
                  Times[ 
                    Rational[ 1, 2], 
                    Plus[ -11, 
                      Power[ 1201, 
                        Rational[ 1, 2]]], T]]]], 
            Subscript[ Y, 2][ T]]], 
        Times[ 
          Power[ E, 
            Times[ 
              Rational[ -8, 3], T]], 
          Subscript[ Y, 3][ T]]}],
      Editable->False]], "Print",
  CellLabel->"From In[13]:="],

Cell[BoxData[
    InterpretationBox[\("After simplification \
"\[InvisibleSpace]{\(\(1\/1201\)\((\[ExponentialE]\^\(\(\(-11\)\ T\)/2\)\ \
\((1201\ Cosh[\(\@1201\ T\)\/2]\ Y\_1[
                      T] + \@1201\ Sinh[\(\@1201\ T\)\/2]\ \((\(-9\)\ Y\_1[
                            T] + 20\ Y\_2[
                            T])\))\))\)\), \[ExponentialE]\^\(\(\(-11\)\ \
T\)/2\)\ Cosh[\(\@1201\ T\)\/2]\ Y\_2[
                T] + \(\[ExponentialE]\^\(\(\(-11\)\ T\)/2\)\ Sinh[\(\@1201\ \
T\)\/2]\ \((56\ Y\_1[T] + 9\ Y\_2[T])\)\)\/\@1201, \
\[ExponentialE]\^\(\(\(-8\)\ T\)/3\)\ Y\_3[T]}\),
      SequenceForm[ "After simplification ", {
        Times[ 
          Rational[ 1, 1201], 
          Power[ E, 
            Times[ 
              Rational[ -11, 2], T]], 
          Plus[ 
            Times[ 1201, 
              Cosh[ 
                Times[ 
                  Rational[ 1, 2], 
                  Power[ 1201, 
                    Rational[ 1, 2]], T]], 
              Subscript[ Y, 1][ T]], 
            Times[ 
              Power[ 1201, 
                Rational[ 1, 2]], 
              Sinh[ 
                Times[ 
                  Rational[ 1, 2], 
                  Power[ 1201, 
                    Rational[ 1, 2]], T]], 
              Plus[ 
                Times[ -9, 
                  Subscript[ Y, 1][ T]], 
                Times[ 20, 
                  Subscript[ Y, 2][ T]]]]]], 
        Plus[ 
          Times[ 
            Power[ E, 
              Times[ 
                Rational[ -11, 2], T]], 
            Cosh[ 
              Times[ 
                Rational[ 1, 2], 
                Power[ 1201, 
                  Rational[ 1, 2]], T]], 
            Subscript[ Y, 2][ T]], 
          Times[ 
            Power[ 1201, 
              Rational[ -1, 2]], 
            Power[ E, 
              Times[ 
                Rational[ -11, 2], T]], 
            Sinh[ 
              Times[ 
                Rational[ 1, 2], 
                Power[ 1201, 
                  Rational[ 1, 2]], T]], 
            Plus[ 
              Times[ 56, 
                Subscript[ Y, 1][ T]], 
              Times[ 9, 
                Subscript[ Y, 2][ T]]]]], 
        Times[ 
          Power[ E, 
            Times[ 
              Rational[ -8, 3], T]], 
          Subscript[ Y, 3][ T]]}],
      Editable->False]], "Print",
  CellLabel->"From In[13]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_3[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[13]="]
}, Open  ]],

Cell["\<\
The simplification is performed only once during the initialization \
phase that constructs the data object for the numerical integration method.\
\
\>", "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
  CellTags->"s:3"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {
            StyleBox["SimplificationFunction",
              "MR"], 
            StyleBox["None",
              "MR"], Cell[TextData[{
              "function to use in simplifying the result of ",
              StyleBox["DSolve", "MR"]
            }], "Text"]}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  "Options of the method ",
  StyleBox["LocallyExact", "MR"],
  ". "
}], "Caption"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.0 for X",
ScreenRectangle->{{0, 1600}, {0, 1200}},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{650, 500},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"LocallyExact",
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
Magnification->1,
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "c:1"->{
    Cell[1776, 53, 50, 1, 87, "Chapter",
      CellTags->"c:1"]},
  "s:1"->{
    Cell[1851, 58, 50, 1, 67, "Section",
      CellTags->"s:1"]},
  "s:2"->{
    Cell[3411, 107, 46, 1, 67, "Section",
      CellTags->"s:2"]},
  "s:3"->{
    Cell[90830, 2549, 52, 1, 67, "Section",
      CellTags->"s:3"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"c:1", 92308, 2597},
  {"s:1", 92385, 2600},
  {"s:2", 92462, 2603},
  {"s:3", 92540, 2606}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 50, 1, 87, "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{
Cell[1851, 58, 50, 1, 67, "Section",
  CellTags->"s:1"],
Cell[1904, 61, 217, 5, 40, "Text"],
Cell[2124, 68, 713, 17, 72, "Text"],
Cell[2840, 87, 315, 8, 56, "Text"],
Cell[3158, 97, 216, 5, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[3411, 107, 46, 1, 67, "Section",
  CellTags->"s:2"],
Cell[3460, 110, 61, 0, 24, "Text"],
Cell[3524, 112, 113, 2, 26, "Input"],

Cell[CellGroupData[{
Cell[3662, 118, 41, 0, 41, "Subsection"],
Cell[3706, 120, 260, 6, 40, "Text"],

Cell[CellGroupData[{
Cell[3991, 130, 511, 12, 106, "Input"],
Cell[4505, 144, 392, 11, 41, "Output"]
}, Open  ]],
Cell[4912, 158, 56, 0, 24, "Text"],

Cell[CellGroupData[{
Cell[4993, 162, 134, 3, 26, "Input"],
Cell[5130, 167, 19520, 508, 294, 4352, 315, "GraphicsData", "PostScript", \
"Graphics"],
Cell[24653, 677, 154, 4, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[24856, 687, 41, 0, 41, "Subsection"],
Cell[24900, 689, 152, 4, 24, "Text"],
Cell[25055, 695, 424, 8, 72, "Text"],
Cell[25482, 705, 135, 3, 24, "Text"],

Cell[CellGroupData[{
Cell[25642, 712, 110, 2, 26, "Input"],
Cell[25755, 716, 52916, 1472, 184, 15588, 1006, "GraphicsData", "PostScript", \
"Graphics"],
Cell[78674, 2190, 154, 4, 25, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[78877, 2200, 36, 0, 41, "Subsection"],
Cell[78916, 2202, 228, 8, 40, "Text"],
Cell[79147, 2212, 184, 4, 40, "Text"],
Cell[79334, 2218, 525, 10, 74, "Input"],
Cell[79862, 2230, 106, 4, 24, "Text"],

Cell[CellGroupData[{
Cell[79993, 2238, 78, 2, 26, "Input"],
Cell[80074, 2242, 1200, 21, 267, "Output"]
}, Open  ]],
Cell[81289, 2266, 204, 5, 40, "Text"],
Cell[81496, 2273, 423, 10, 122, "Input"],
Cell[81922, 2285, 127, 4, 24, "Text"],

Cell[CellGroupData[{
Cell[82074, 2293, 238, 5, 42, "Input"],
Cell[82315, 2300, 5148, 141, 255, "Print"],
Cell[87466, 2443, 2368, 71, 176, "Print"],
Cell[89837, 2516, 758, 20, 57, "Output"]
}, Open  ]],
Cell[90610, 2539, 171, 4, 40, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[90830, 2549, 52, 1, 67, "Section",
  CellTags->"s:3"],
Cell[90885, 2552, 530, 13, 82, "DefinitionBox3Col"],
Cell[91418, 2567, 99, 4, 16, "Caption"]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

