(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.1'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    259902,       6237]*)
(*NotebookOutlinePosition[    261430,       6293]*)
(*  CellTagsIndexPosition[    261222,       6282]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["OrthogonalProjection", "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"s:1"],

Cell["Consider the matrix differential equation:", "Text"],

Cell[BoxData[
    \(TraditionalForm\`\(y\^\[Prime]\)(t) = f(t, y(t)), 
    t > 0, \)], "DisplayFormula"],

Cell[TextData[{
  "where the initial value ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(y\_0\), "=", " ", 
          RowBox[{\(y(0)\), Cell[""]}]}], TraditionalForm]]],
  "\[Element] ",
  Cell[BoxData[
      \(TraditionalForm\`\[DoubleStruckCapitalR]\^\(m\[Times]p\)\)]],
  " is given. Assume that ",
  Cell[BoxData[
      \(TraditionalForm\`\(y\_0\^T\) y\_0\  = \ I\)]],
  ", that the solution has the property of preserving orthonormality, ",
  Cell[BoxData[
      \(TraditionalForm\`\(y(t)\)\^T\ \(y(t)\)\  = \ I\)]],
  ", and that it has full rank for all t \[GreaterEqual] 0."
}], "Text"],

Cell[TextData[{
  "From a numerical perspective, a key issue is how to numerically integrate \
an orthogonal matrix differential system in such a way that the numerical \
solution remains orthogonal. There are several strategies that are possible. \
One approach, suggested in [",
  ButtonBox["DRV94",
    ButtonData:>{"References.nb", "DRV94"},
    ButtonStyle->"Hyperlink"],
  "], is to use an implicit Runge-Kutta method (such as the Gauss scheme). \
Some alternative strategies are described in [",
  ButtonBox["DV99",
    ButtonData:>{"References.nb", "DV99"},
    ButtonStyle->"Hyperlink"],
  "] and [",
  ButtonBox["DL01",
    ButtonData:>{"References.nb", "DL01"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell["\<\
The approach which we will take up here is to use any reasonable \
numerical integration method and then post-process using a projective \
procedure at the end of each integration step.\
\>", "Text"],

Cell[TextData[{
  "An important feature of our implementation is that the basic integration \
method can be any built-in numerical method, or even a user-defined \
procedure. In the following examples an explicit Runge-Kutta method is used \
for the basic time stepping. However, if we require greater accuracy we could \
easily use an extrapolation method, for example, by simply setting the \
appropriate ",
  StyleBox["Method", "MR"],
  " option."
}], "Text"],

Cell[CellGroupData[{

Cell["Projection step", "Subsection"],

Cell[TextData[{
  "At the end of each numerical integration step we need to transform the \
approximate solution matrix of the differential system to obtain an \
orthogonal matrix. This can be carried out in several ways (see for example \
[",
  ButtonBox["DRV94",
    ButtonData:>{"References.nb", "DRV94"},
    ButtonStyle->"Hyperlink"],
  "] and [",
  ButtonBox["H97",
    ButtonData:>{"References.nb", "H97"},
    ButtonStyle->"Hyperlink"],
  "]):"
}], "Text"],

Cell["Newton or Schulz iteration", "BulletedList"],

Cell["QR decomposition", "BulletedList"],

Cell["Singular value decomposition", "BulletedList"],

Cell["\<\
The Newton and Schulz methods are quadratically convergent, and the \
number of iterations may vary depending on the error tolerances used in the \
numerical integration. One or two iterations are usually sufficient for \
convergence to the orthonormal polar factor (see below) in IEEE \
double-precision arithmetic.\
\>", "Text"],

Cell["\<\
QR decomposition is cheaper than singular value decomposition \
(roughly by a factor of two), but it does not give the closest possible \
projection.\
\>", "Text"],

Cell[TextData[{
  StyleBox["Definition",
    FontWeight->"Bold"],
  StyleBox[" (Thin singular value decomposition ",
    FontVariations->{"CompatibilityType"->0}],
  "[",
  ButtonBox["GV96",
    ButtonData:>{"References.nb", "GV96"},
    ButtonStyle->"Hyperlink"],
  "]",
  StyleBox["):",
    FontVariations->{"CompatibilityType"->0}],
  " Given a matrix ",
  Cell[BoxData[
      \(TraditionalForm\`A \[Element] \[DoubleStruckCapitalR]\^\(m\[Times]p\)\
\)]],
  " with ",
  StyleBox["m",
    FontSlant->"Italic"],
  " \[GreaterEqual] ",
  StyleBox["p",
    FontSlant->"Italic"],
  " there exist two matrices U \[Element] ",
  Cell[BoxData[
      \(TraditionalForm\`\[DoubleStruckCapitalR]\^\(m\[Times]p\)\)]],
  " and ",
  StyleBox["V",
    FontSlant->"Italic"],
  " \[Element] ",
  Cell[BoxData[
      \(TraditionalForm\`\[DoubleStruckCapitalR]\^\(p\[Times]p\)\)]],
  " such that ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(U\^T\)\(\ \)\(A\)\(\ \)\(V\)\(\ \)\)\)]],
  " is the diagonal matrix of singular values of ",
  StyleBox["A",
    FontSlant->"Italic"],
  ", \[CapitalSigma] = ",
  Cell[BoxData[
      \(TraditionalForm\`diag(\[Sigma]\_1, \[Ellipsis], \[Sigma]\_p)\)]],
  "\[Element] ",
  Cell[BoxData[
      \(TraditionalForm\`\[DoubleStruckCapitalR]\^\(p\[Times]p\)\)]],
  ", where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Sigma]\_1 \[GreaterEqual] \[CenterEllipsis] \
\[GreaterEqual] \[Sigma]\_p \[GreaterEqual] 0\)]],
  ". ",
  StyleBox["U",
    FontSlant->"Italic"],
  " has orthonormal columns and ",
  StyleBox["V",
    FontSlant->"Italic"],
  " is orthogonal."
}], "Text"],

Cell[TextData[{
  StyleBox["Definition",
    FontWeight->"Bold"],
  StyleBox[" (Polar decomposition):",
    FontVariations->{"CompatibilityType"->0}],
  " Given a matrix ",
  Cell[BoxData[
      \(TraditionalForm\`A\)]],
  " and its singular value decomposition ",
  Cell[BoxData[
      FormBox[
        RowBox[{"U", " ", "\[CapitalSigma]", " ", 
          FormBox[\(V\^T\),
            "TraditionalForm"]}], TraditionalForm]]],
  ", the polar decomposition of A is given by the product of two matrices ",
  StyleBox["Z",
    FontSlant->"Italic"],
  " and ",
  StyleBox["P",
    FontSlant->"Italic"],
  " where ",
  Cell[BoxData[
      \(TraditionalForm\`Z = U\ V\^T\)]],
  " and ",
  StyleBox["P",
    FontSlant->"Italic"],
  " = ",
  Cell[BoxData[
      \(TraditionalForm\`V\ \[CapitalSigma]\ V\^T\)]],
  ". ",
  StyleBox["Z",
    FontSlant->"Italic"],
  " has orthonormal columns and ",
  StyleBox["P",
    FontSlant->"Italic"],
  " is symmetric positive semidefinite."
}], "Text"],

Cell["\<\
The orthonormal polar factor Z of A is the matrix that \
solves:\
\>", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        UnderscriptBox["min", 
          RowBox[{"Z", " ", "\[Element]", " ", 
            FormBox[\(\[DoubleStruckCapitalR]\^\(m\[Times]p\)\),
              "TraditionalForm"]}]], 
        RowBox[{"{", 
          RowBox[{
            RowBox[{\(\(||\)\(A - Z\)\(||\)\), " ", ":", " ", 
              RowBox[{
                FormBox[\(Z\^T\),
                  "TraditionalForm"], "Z"}]}], "=", "I"}], "}"}]}], 
      TraditionalForm]], "DisplayFormula"],

Cell[TextData[{
  "for the 2 and Frobenius norms [",
  ButtonBox["H96",
    ButtonData:>{"References.nb", "H96"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Schulz iteration", "Subsection"],

Cell[TextData[{
  "We have chosen an approach based on the Schulz iteration, which works \
directly for ",
  StyleBox["m",
    FontSlant->"Italic"],
  " >= ",
  StyleBox["p",
    FontSlant->"Italic"],
  ". In contrast Newton iteration for ",
  StyleBox["m",
    FontSlant->"Italic"],
  " > ",
  StyleBox["p",
    FontSlant->"Italic"],
  " needs to be preceded by QR decomposition."
}], "Text"],

Cell["\<\
Comparison with direct computation based on the singular value \
decomposition is also given.\
\>", "Text"],

Cell["The Schulz iteration is given by:", "Text"],

Cell[BoxData[
    \(TraditionalForm\`Y\_\(i + 1\) = 
      Y\_i + \(\(Y\_i\)(I - Y\_i\%T\ Y\_i)\)/2, \ 
    Y\_0 = \(\(A\)\(.\)\)\)], "NumberedEquation",
  CellTags->"SchulzIteration"],

Cell[TextData[{
  "The Schulz iteration has an arithmetic operation count per iteration of ",
  Cell[BoxData[
      \(TraditionalForm\`2 \( m\^2\) p + 2  m\ p\^2\)]],
  " floating-point operations, but is rich in matrix multiplication [",
  ButtonBox["H97",
    ButtonData:>{"References.nb", "H97"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell[TextData[{
  "In a practical implementation, gemm level 3 BLAS of LAPACK [",
  ButtonBox["LAPACK99",
    ButtonData:>{"References.nb", "LAPACK99"},
    ButtonStyle->"Hyperlink"],
  "] can be used in conjunction with architecture-specific optimizations via \
the Automatically Tuned Linear Algebra Software [",
  ButtonBox["ATLAS00",
    ButtonData:>{"References.nb", "ATALS00"},
    ButtonStyle->"Hyperlink"],
  "]. Such considerations mean that the arithmetic operation count of the \
Schulz iteration is not necessarily an accurate reflection of the observed \
computational cost. A useful bound on the departure from orthonormality of A \
in is [",
  ButtonBox["H89",
    ButtonData:>{"References.nb", "H89"},
    ButtonStyle->"Hyperlink"],
  "]: ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(||\)\(A\^T\ A\  - \ I\)\( || \_F\)\)\)]],
  ". Comparison with the Schulz iteration gives the stopping criterion ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(||\)\(A\^T\ A\  - \ 
          I\)\( || \_F\)\(\(<\)\(\[Tau]\)\)\)\)]],
  " for some tolerance \[Tau]."
}], "Text"],

Cell[CellGroupData[{

Cell["Standard formulation", "Subsubsection"],

Cell[TextData[{
  "Assume that an initial value ",
  Cell[BoxData[
      \(TraditionalForm\`y\_n\)]],
  " for the current solution of the ODE is given, together with a solution ",
  Cell[BoxData[
      \(TraditionalForm\`y\_\(n + 1\) = y\_n + \[CapitalDelta]y\_n\)]],
  " from a one-step numerical integration method. Assume that an absolute \
tolerance \[Tau] for controlling the Schulz iteration is also prescribed."
}], "Text"],

Cell["The following algorithm can be used for implementation.", "Text"],

Cell[TextData[{
  "Step 1.",
  " ",
  "Set ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_0 = y\_\(n + 1\)\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`i = 0\)]],
  ".\nStep 2. Compute ",
  Cell[BoxData[
      \(TraditionalForm\`E\  = \ I\  - \ Y\_i\%T\ Y\_i\)]],
  ".\nStep 3. Compute ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_\(i + 1\) = Y\_i + \(Y\_i\) E/2\)]],
  ".\nStep 4. If ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(||\)\(E\)\( || \_F\)\(\(\[LessEqual]\)\(\[Tau]\)\
\)\)\)]],
  " or ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(i = imax\)\(,\)\)\)]],
  " then return ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_\(i + 1\)\)]],
  ".\nStep 5. Set ",
  Cell[BoxData[
      \(TraditionalForm\`i = \ i\  + \ 1\)]],
  " and go to Step 2."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Increment formulation", "Subsubsection"],

Cell[TextData[{
  StyleBox["NDSolve", "MR"],
  " uses compensated summation to reduce the effect of rounding errors made \
by repeatedly adding the contribution of small quantities ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalDelta]y\_n\)]],
  "to ",
  Cell[BoxData[
      \(TraditionalForm\`y\_n\)]],
  " at each integration step [",
  ButtonBox["H96",
    ButtonData:>{"References.nb", "H96"},
    ButtonStyle->"Hyperlink"],
  "]. Therefore, the increment ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalDelta]y\_n\)]],
  " is returned by the base integrator."
}], "Text"],

Cell[TextData[{
  "An appropriate orthogonal correction ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalDelta]Y\_i\)]],
  " for the projective iteration can be determined using the following \
algorithm."
}], "Text"],

Cell[TextData[{
  "Step 1. Set ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalDelta]Y\_0 = 0\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`i = 0\)]],
  ".\nStep 2. Set ",
  Cell[BoxData[
      \(TraditionalForm\`Y\_i = \[CapitalDelta]Y\_i + y\_\(n + 1\)\)]],
  ".\nStep 3. Compute ",
  Cell[BoxData[
      \(TraditionalForm\`E\  = \ I\  - \ Y\_i\%T\ Y\_i\)]],
  ".\nStep 4. Compute ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalDelta]Y\_\(i + 1\) = \[CapitalDelta]Y\_i + \
\(Y\_i\) E/2\)]],
  ".\nStep 5. If ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(||\)\(E\)\( || \_F\)\(\(\[LessEqual]\)\(\[Tau]\)\
\)\)\)]],
  " or ",
  Cell[BoxData[
      \(TraditionalForm\`i = imax\)]],
  ", then return ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalDelta]Y\_\(i + 1\) + \
\[CapitalDelta]y\_n\)]],
  ".\nStep 6. Set ",
  Cell[BoxData[
      \(TraditionalForm\`i = \ i\  + \ 1\)]],
  " and go to Step 2."
}], "Text"],

Cell[TextData[{
  "This modified algorithm is used in ",
  StyleBox["OrthogonalProjection ", "MR"],
  "and shows an advantage of using an iterative process over a direct \
process, since it is not obvious how an orthogonal correction can be derived \
for direct methods."
}], "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
  CellTags->"s:2"],

Cell[CellGroupData[{

Cell["Orthogonal error measurement", "Subsection"],

Cell[TextData[{
  "A function to compute the Frobenius norm ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(||\)\(A\)\( || \_F\)\)\)]],
  " of a matrix ",
  StyleBox["A",
    FontSlant->"Italic"],
  " can be defined in terms of the ",
  StyleBox["Norm", "MR"],
  " function as follows."
}], "Text"],

Cell[BoxData[
    \(FrobeniusNorm[a_?MatrixQ] := \ Norm[a, \ Frobenius]; \)], "Input",
  CellLabel->"In[1]:="],

Cell[TextData[{
  "An upper bound on the departure from orthonormality of ",
  StyleBox["A",
    FontSlant->"Italic"],
  " can then be measured using this function [",
  ButtonBox["H97",
    ButtonData:>{"References.nb", "H97"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell[BoxData[
    \(OrthogonalError[a_?MatrixQ] := \ 
      FrobeniusNorm[
        Transpose[a] . a\  - \ 
          IdentityMatrix[Last[Dimensions[a]]]]; \)], "Input",
  CellLabel->"In[2]:="],

Cell["\<\
This defines the utility function for visualizing the orthogonal \
error during a numerical integration.\
\>", "Text"],

Cell[BoxData[
    \(\(\( (*\ 
      Utility\ function\ for\ extracting\ a\ list\ of\ values\ of\ the\ \
independent\ variable\ at\ which\ the\ integration\ method\ has\ sampled\ \
*) \)\(\[IndentingNewLine]\)\(\[IndentingNewLine]\)\(\(TimeData[{v_?
              VectorQ, ___?VectorQ}] := 
        TimeData[
          v]; \)\[IndentingNewLine]\[IndentingNewLine]\(TimeData[{if : \
\((InterpolatingFunction[__])\)[_], ___}] := \[IndentingNewLine]Part[if, 0, 
          3, 1]; \)\)\)\)], "Input",
  CellLabel->"In[3]:="],

Cell[BoxData[
    \(\(\( (*\ 
      Utility\ function\ for\ plotting\ the\ orthogonal\ error\ in\ a\ \
numerical\ integration\ \
*) \)\(\[IndentingNewLine]\)\(\[IndentingNewLine]\)\(OrthogonalErrorPlot[
        sol_] := \[IndentingNewLine]Module[{errdata, \ samples, \ 
          soldata}, \[IndentingNewLine] (*\ 
          Form\ a\ list\ of\ times\ at\ which\ the\ method\ is\ invoked\ \
*) \[IndentingNewLine]samples\  = \ TimeData[sol]; \[IndentingNewLine] (*\ 
          Form\ a\ list\ of\ solutions\ at\ the\ integration\ times\ \
*) \[IndentingNewLine]soldata = 
          Map[\((sol /. t \[Rule] #)\) &, samples]; \[IndentingNewLine] (*\ 
          Form\ a\ list\ of\ the\ orthogonal\ errors\ \
*) \[IndentingNewLine]errdata\  = \ 
          Map[OrthogonalError, \ soldata]; \[IndentingNewLine]ListPlot[
          Transpose[{samples, errdata}], \ PlotJoined \[Rule] True, 
          Frame \[Rule] True, 
          PlotLabel \[Rule] \*"\"\<Orthogonal error ||\!\(Y\^T\)Y - I\!\(\( \
|| \_F\)\) vs time\>\""\[IndentingNewLine]]; \[IndentingNewLine]]; \)\)\)], \
"Input",
  CellLabel->"In[5]:="]
}, Open  ]],

Cell[CellGroupData[{

Cell["Square systems", "Subsection"],

Cell[TextData[{
  "This example concerns the solution of a matrix differential system on the \
orthogonal group ",
  Cell[BoxData[
      \(TraditionalForm\`\(O\_3\)(\[DoubleStruckCapitalR])\)]],
  " (see ",
  "[",
  ButtonBox["Z98",
    ButtonData:>{"References.nb", "Z98"},
    ButtonStyle->"Hyperlink"],
  "]",
  ")."
}], "Text"],

Cell["The matrix differential system is given by", "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {\(Y\^\[Prime]\), "=", \(\(F(Y)\)\ Y\)},
          {" ", "=", \(\((A\  + \ \((I - \ Y\ Y\^T)\))\)\ Y\)}
          }], TraditionalForm]], "DisplayFormula"],

Cell["with", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{"A", " ", "=", " ", 
        TagBox[
          RowBox[{"(", "\[NoBreak]", GridBox[{
                {"0", \(-1\), "1"},
                {"1", "0", "1"},
                {\(-1\), \(-1\), "0"}
                }], "\[NoBreak]", ")"}],
          Function[ BoxForm`e$, 
            MatrixForm[ BoxForm`e$]]]}], TraditionalForm]], "DisplayFormula"],

Cell["and", "Text"],

Cell[BoxData[
    \(TraditionalForm\`Y\_0 = \(\(I\_3\)\(.\)\)\)], "DisplayFormula"],

Cell["The solution evolves as:", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{\(Y(t)\), " ", "=", " ", 
        TagBox[\(\(exp[t\ A]\)\(.\)\),
          Function[ BoxForm`e$, 
            MatrixForm[ BoxForm`e$]]]}], TraditionalForm]], "DisplayFormula"],

Cell[TextData[{
  "The eigenvalues of Y(t) are ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_1 = 1\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_2 = exp(t\ i\ \@3)\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_3 = exp(\(-\ t\)\ i\ \@3)\)]],
  ". Thus as ",
  StyleBox["t",
    FontSlant->"Italic"],
  " approaches ",
  Cell[BoxData[
      \(TraditionalForm\`\[Pi]/\@3\)]],
  ", two of the eigenvalues of ",
  StyleBox["Y",
    FontSlant->"Italic"],
  "(",
  StyleBox["t",
    FontSlant->"Italic"],
  ") approach -1. The numerical integration is carried out on the interval \
[0, 2]."
}], "Text"],

Cell[BoxData[{
    RowBox[{\(n\  = \ 3; \), "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
    RowBox[{
      RowBox[{
        RowBox[{"A", " ", "=", " ", 
          TagBox[
            RowBox[{"(", "\[NoBreak]", GridBox[{
                  {"0", \(-1\), "1"},
                  {"1", "0", "1"},
                  {\(-1\), \(-1\), "0"}
                  }], "\[NoBreak]", ")"}],
            Function[ BoxForm`e$, 
              MatrixForm[ BoxForm`e$]]]}], ";"}], 
      "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
    RowBox[{\(Y = Table[\(y[i, j]\)[t], {i, n}, {j, n}]; \), 
      "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
    RowBox[{\(F\  = A\  + \((\ IdentityMatrix[n]\  - Transpose[Y] . Y)\); \), 
      "\[IndentingNewLine]", 
      "\[IndentingNewLine]", \( (*\ Vector\ differential\ system\ *) \), 
      "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
    RowBox[{\(system = Thread[Flatten[D[Y, t]] \[Equal] Flatten[F . Y]]; \), 
      "\[IndentingNewLine]", 
      "\[IndentingNewLine]", \( (*\ Vector\ initial\ conditions\ *) \), 
      "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
    RowBox[{\(ics = 
        Thread[Flatten[\((Y /. t \[Rule] 0)\)] \[Equal] 
            Flatten[IdentityMatrix[Length[Y]]]]; \), 
      "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
    RowBox[{\(eqs = {system, ics}; \), 
      "\[IndentingNewLine]"}], "\[IndentingNewLine]", 
    RowBox[{\(vars\  = \ Flatten[Y]; \), 
      "\[IndentingNewLine]"}], "\[IndentingNewLine]", \(time = {t, 0, 
        2}; \)}], "Input",
  CellLabel->"In[6]:="],

Cell[TextData[{
  "This computes the solution using an explicit Runge-Kutta method. The \
appropriate initial step size and method order are selected automatically, \
and the step size may vary throughout the integration interval, which is \
chosen in order to satisfy local relative and absolute error tolerances. \
Alternatively, the order of the method could be specified by using a ",
  StyleBox["Method", "MR"],
  " option."
}], "Text"],

Cell[BoxData[
    \(solerk = 
      NDSolve[eqs, vars, time, 
        Method \[Rule] "\<ExplicitRungeKutta\>"]; \)], "Input",
  CellLabel->"In[15]:="],

Cell["\<\
This computes the orthogonal error, or absolute deviation from the \
orthogonal manifold, as the integration progresses. The error is of the order \
of the local accuracy of the numerical method.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(solerk\  = \ 
      Y /. First[solerk]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(OrthogonalErrorPlot[solerk]; \)}], "Input",
  CellLabel->"In[16]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.47619 -0.762226 2.34063e+09 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.2619 -0.0125 -9 -9 ]
[.2619 -0.0125 9 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.7381 -0.0125 -9 -9 ]
[.7381 -0.0125 9 0 ]
[.97619 -0.0125 -3 -9 ]
[.97619 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .05699 -49.5 -6.25 ]
[-0.0125 .05699 0 6.25 ]
[-0.0125 .17402 -37.5 -6.25 ]
[-0.0125 .17402 0 6.25 ]
[-0.0125 .29106 -49.5 -6.25 ]
[-0.0125 .29106 0 6.25 ]
[-0.0125 .40809 -37.5 -6.25 ]
[-0.0125 .40809 0 6.25 ]
[-0.0125 .52512 -49.5 -6.25 ]
[-0.0125 .52512 0 6.25 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[.5 .63053 -113.125 0 ]
[.5 .63053 113.125 13.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.2619 0 m
.2619 .00625 L
s
[(0.5)] .2619 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(1)] .5 -0.0125 0 1 Mshowa
.7381 0 m
.7381 .00625 L
s
[(1.5)] .7381 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(2)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07143 0 m
.07143 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.16667 0 m
.16667 .00375 L
s
.21429 0 m
.21429 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.35714 0 m
.35714 .00375 L
s
.40476 0 m
.40476 .00375 L
s
.45238 0 m
.45238 .00375 L
s
.54762 0 m
.54762 .00375 L
s
.59524 0 m
.59524 .00375 L
s
.64286 0 m
.64286 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.78571 0 m
.78571 .00375 L
s
.83333 0 m
.83333 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .05699 m
.00625 .05699 L
s
gsave
-0.0125 .05699 -49.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(3.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
0 .17402 m
.00625 .17402 L
s
gsave
-0.0125 .17402 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
0 .29106 m
.00625 .29106 L
s
gsave
-0.0125 .29106 -49.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(4.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
0 .40809 m
.00625 .40809 L
s
gsave
-0.0125 .40809 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
0 .52512 m
.00625 .52512 L
s
gsave
-0.0125 .52512 -49.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(5.5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-10) show
grestore
.125 Mabswid
0 .0804 m
.00375 .0804 L
s
0 .10381 m
.00375 .10381 L
s
0 .12721 m
.00375 .12721 L
s
0 .15062 m
.00375 .15062 L
s
0 .19743 m
.00375 .19743 L
s
0 .22084 m
.00375 .22084 L
s
0 .24424 m
.00375 .24424 L
s
0 .26765 m
.00375 .26765 L
s
0 .31446 m
.00375 .31446 L
s
0 .33787 m
.00375 .33787 L
s
0 .36127 m
.00375 .36127 L
s
0 .38468 m
.00375 .38468 L
s
0 .43149 m
.00375 .43149 L
s
0 .4549 m
.00375 .4549 L
s
0 .47831 m
.00375 .47831 L
s
0 .50171 m
.00375 .50171 L
s
0 .03359 m
.00375 .03359 L
s
0 .01018 m
.00375 .01018 L
s
0 .54852 m
.00375 .54852 L
s
0 .57193 m
.00375 .57193 L
s
0 .59534 m
.00375 .59534 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.02381 .61178 m
.02381 .61803 L
s
.2619 .61178 m
.2619 .61803 L
s
.5 .61178 m
.5 .61803 L
s
.7381 .61178 m
.7381 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.07143 .61428 m
.07143 .61803 L
s
.11905 .61428 m
.11905 .61803 L
s
.16667 .61428 m
.16667 .61803 L
s
.21429 .61428 m
.21429 .61803 L
s
.30952 .61428 m
.30952 .61803 L
s
.35714 .61428 m
.35714 .61803 L
s
.40476 .61428 m
.40476 .61803 L
s
.45238 .61428 m
.45238 .61803 L
s
.54762 .61428 m
.54762 .61803 L
s
.59524 .61428 m
.59524 .61803 L
s
.64286 .61428 m
.64286 .61803 L
s
.69048 .61428 m
.69048 .61803 L
s
.78571 .61428 m
.78571 .61803 L
s
.83333 .61428 m
.83333 .61803 L
s
.88095 .61428 m
.88095 .61803 L
s
.92857 .61428 m
.92857 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .05699 m
1 .05699 L
s
.99375 .17402 m
1 .17402 L
s
.99375 .29106 m
1 .29106 L
s
.99375 .40809 m
1 .40809 L
s
.99375 .52512 m
1 .52512 L
s
.125 Mabswid
.99625 .0804 m
1 .0804 L
s
.99625 .10381 m
1 .10381 L
s
.99625 .12721 m
1 .12721 L
s
.99625 .15062 m
1 .15062 L
s
.99625 .19743 m
1 .19743 L
s
.99625 .22084 m
1 .22084 L
s
.99625 .24424 m
1 .24424 L
s
.99625 .26765 m
1 .26765 L
s
.99625 .31446 m
1 .31446 L
s
.99625 .33787 m
1 .33787 L
s
.99625 .36127 m
1 .36127 L
s
.99625 .38468 m
1 .38468 L
s
.99625 .43149 m
1 .43149 L
s
.99625 .4549 m
1 .4549 L
s
.99625 .47831 m
1 .47831 L
s
.99625 .50171 m
1 .50171 L
s
.99625 .03359 m
1 .03359 L
s
.99625 .01018 m
1 .01018 L
s
.99625 .54852 m
1 .54852 L
s
.99625 .57193 m
1 .57193 L
s
.99625 .59534 m
1 .59534 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
gsave
.5 .63053 -174.125 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.5 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 12.812 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Orthogonal) show
129.000 12.812 moveto
(error) show
165.000 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
177.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
183.000 9.000 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(T) show
189.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
201.125 12.812 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
213.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(I) show
219.125 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
231.125 16.500 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(F) show
243.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(vs) show
261.250 12.812 moveto
(time) show
285.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.05366 0 m
.07281 .48916 L
.10075 .36694 L
.13955 .40577 L
.17377 .32844 L
.21124 .32905 L
.2479 .30819 L
.28579 .32434 L
.32382 .34308 L
.36256 .38316 L
.40174 .43443 L
.44103 .48246 L
.48007 .51327 L
.51893 .53248 L
.55751 .53861 L
.59584 .53499 L
.6339 .52343 L
.67197 .51378 L
.71036 .51599 L
.74904 .52829 L
.78772 .53874 L
.82661 .5552 L
.86567 .57558 L
.90499 .60332 L
.94059 .52104 L
.97619 .45019 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[16]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00>@3oool2000002l0oooo0P0000040?ooo`030000
003oool0oooo0080oooo0`00000_0?ooo`@00000;P3oool4000000<0oooo00<000000?ooo`3oool0
0P3oool3000002l0oooo100000050?ooo`00>03oool010000000oooo0?ooo`00000]0?ooo`040000
003oool0oooo000000/0oooo00<000000?ooo`3oool0;P3oool00`000000oooo0?ooo`0_0?ooo`03
0000003oool0oooo00X0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`060?ooo`00
>03oool010000000oooo0?ooo`00000]0?ooo`040000003oool0oooo000000/0oooo00<000000?oo
o`3oool0;P3oool00`000000oooo0?ooo`0_0?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool0;@3oool00`000000oooo0?ooo`050?ooo`00>03oool010000000oooo0?ooo`00000]0?oo
o`040000003oool0oooo000000P0oooo0`00000a0?ooo`030000003oool0oooo02l0oooo00<00000
0?ooo`3oool01`3oool300000340oooo00<000000?ooo`3oool0103oool003P0oooo00@000000?oo
o`3oool00000;@3oool010000000oooo0?ooo`0000080?ooo`030000003oool0oooo0340oooo00<0
00000?ooo`3oool0;`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0380oooo00<0
00000?ooo`3oool00`3oool003P0oooo00@000000?ooo`3oool00000;@3oool010000000oooo0?oo
o`0000080?ooo`030000003oool0oooo02l0oooo0`00000_0?ooo`<000002@3oool00`000000oooo
0?ooo`0_0?ooo`040000003oool0oooo000000D0oooo000i0?ooo`800000;`3oool2000000T0oooo
1000000`0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool01`3oool4000002l0oooo
0P0000060?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00=03oooo[00000040
oooo000d0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0103oool01@000000oooo
0?ooo`3oool0000000X0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool00P3oool100000040oooo000d0?ooo`030000
003oool0oooo00<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0]0?ooo`030000
003oool0oooo03D0oooo00<000000?ooo`3oool0=@3oool00`000000oooo0?ooo`0e0?ooo`030000
003oool0oooo0080oooo0@0000010?ooo`00=03oool2000000/0oooo00<000000?ooo`3oool0f@3o
ool200000040oooo000d0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0fP3oool1
00000040oooo000d0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0fP3oool10000
0040oooo000d0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0fP3oool100000040
oooo000d0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0fP3oool100000040oooo
000d0?ooo`8000002`3oool00`000000oooo0?ooo`3I0?ooo`8000000@3oool003@0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`3J0?ooo`4000000@3oool00080oooo0P0000040?oo
o`030000003oool0oooo0080oooo0`0000030?ooo`050000003oool0oooo0?ooo`0000001P3oool4
000000<0oooo0P00000=0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0fP3oool1
00000040oooo00001@3oool000000?ooo`3oool0000000/0oooo00L000000?ooo`3oool0oooo0000
003oool0000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`3J0?ooo`4000000@3oool000@0oooo
00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00P0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3o
ool02P3oool00`000000oooo0?ooo`3J0?ooo`4000000@3oool00080oooo0P0000090?ooo`<00000
103oool00`000000oooo000000090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000
oooo0?ooo`0:0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0fP3oool100000040
oooo00040?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00`3oool01@000000oooo
0?ooo`3oool0000000P0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00X0oooo0P00000;0?ooo`030000003oool0oooo0=T0oooo0P0000010?ooo`0000D0oooo0000003o
ool0oooo000000080?ooo`030000003oool0oooo00h0oooo0`0000030?ooo`040000003oool0oooo
000000`0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`3I0?ooo`4000000@3oool0
0080oooo0P0000090?ooo`@000003`3oool00`000000oooo0?ooo`020?ooo`800000203oool30000
0080oooo0P00000<0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`00<03oool01`000000
oooo0?ooo`3oool000000?ooo`0000002`3oool00`000000oooo0?ooo`3I0?ooo`4000000@3oool0
02X0oooo0`0000030?ooo`070000003oool0oooo0?ooo`000000oooo0000000;0?ooo`030000003o
ool0oooo0=T0oooo0@0000010?ooo`00<03oool010000000oooo0?ooo`3oool3000000/0oooo00<0
00000?ooo`3oool0f03oool200000040oooo000_0?ooo`8000000`3oool00`000000oooo0000000;
0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`00<03oool010000000oooo0?ooo`3oool2
000000`0oooo00<000000?ooo`3oool0f@3oool100000040oooo000d0?ooo`030000003oool0oooo
00/0oooo00<000000?ooo`3oool0f@3oool100000040oooo000d0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool0f@3oool100000040oooo000d0?ooo`030000003oool0oooo00/0oooo
00<000000?ooo`3oool0f@3oool100000040oooo000d0?ooo`800000303oool00`000000oooo0?oo
o`3H0?ooo`8000000@3oool003@0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`3I
0?ooo`4000000@3oool003@0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`3I0?oo
o`4000000@3oool003@0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`3I0?ooo`40
00000@3oool003@0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`3I0?ooo`400000
0@3oool003@0oooo0P00000<0?ooo`030000003oool0oooo0=P0oooo0P0000010?ooo`00=03oool0
0`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`00=03oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`00=03oool00`000000
oooo0?ooo`0;0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`00=03oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`00=03oool00`000000oooo0?oo
o`0;0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`00=03oool2000000`0oooo00<00000
0?ooo`3oool0f03oool200000040oooo000d0?ooo`030000003oool0oooo00/0oooo00<000000?oo
o`3oool0f@3oool100000040oooo000?0?ooo`050000003oool0oooo0?ooo`0000000`3oool00`00
0000oooo0?ooo`040?ooo`@000000`3oool2000000d0oooo00<000000?ooo`3oool02`3oool00`00
0000oooo0?ooo`3I0?ooo`4000000@3oool000l0oooo00<000000?ooo`3oool00P3oool00`000000
oooo000000090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00/0oooo00<000000?ooo`3oool0f@3oool100000040oooo000<0?ooo`D0
0000103oool00`000000oooo0?ooo`080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`00
0000oooo0?ooo`0:0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0f@3oool10000
0040oooo000<0?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`0000002@3oool01@00
0000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool2000000`0oooo00<00000
0?ooo`3oool0f03oool200000040oooo000=0?ooo`030000003oool0000000<0oooo00D000000?oo
o`3oool0oooo000000080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0f@3oool100000040oooo000>
0?ooo`8000003P3oool3000000<0oooo00@000000?ooo`3oool000001`3oool300000080oooo0P00
000<0?ooo`030000003oool0oooo0=T0oooo0@0000010?ooo`003`3oool00`000000oooo0?ooo`0>
0?ooo`030000003oool0oooo0080oooo0P0000090?ooo`070000003oool0oooo0?ooo`000000oooo
0000000<0?ooo`030000003oool0oooo0=P0oooo0@0000010?ooo`00:P3oool3000000<0oooo00L0
00000?ooo`3oool0oooo0000003oool0000000`0oooo00<000000?ooo`3oool0f03oool100000040
oooo000`0?ooo`070000003oool0oooo0?ooo`000000oooo0000000<0?ooo`030000003oool0oooo
0=P0oooo0@0000010?ooo`00;`3oool2000000<0oooo0`00000<0?ooo`030000003oool0oooo0=L0
oooo0P0000010?ooo`00<03oool010000000oooo0?ooo`3oool2000000d0oooo00<000000?ooo`3o
ool0f03oool100000040oooo000d0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0
f03oool100000040oooo000d0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0f03o
ool100000040oooo000d0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0f03oool1
00000040oooo000d0?ooo`8000003@3oool00`000000oooo0?ooo`3G0?ooo`8000000@3oool003@0
oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`3H0?ooo`4000000@3oool003@0oooo
00<000000?ooo`3oool0303oool00`000000oooo0?ooo`3H0?ooo`4000000@3oool003@0oooo00<0
00000?ooo`3oool0303oool00`000000oooo0?ooo`3H0?ooo`4000000@3oool003@0oooo00<00000
0?ooo`3oool0303oool00`000000oooo0?ooo`3H0?ooo`4000000@3oool003@0oooo00<000000?oo
o`3oool0303oool00`000000oooo0?ooo`3H0?ooo`4000000@3oool003@0oooo0P00000=0?ooo`03
0000003oool0oooo0=L0oooo0P0000010?ooo`00=03oool00`000000oooo0?ooo`0<0?ooo`030000
003oool0oooo0=P0oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`0<0?ooo`030000003o
ool0oooo0=P0oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo0=P0oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo
0=P0oooo0@0000010?ooo`00=03oool2000000d0oooo00<000000?ooo`3oool0e`3oool200000040
oooo000d0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0f03oool100000040oooo
00030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool3000000<0oooo00D0
00000?ooo`3oool0oooo000000060?ooo`@000000`3oool2000000d0oooo00<000000?ooo`3oool0
303oool00`000000oooo0?ooo`3H0?ooo`4000000@3oool000<0oooo00<000000?ooo`3oool02P3o
ool01`000000oooo0?ooo`3oool000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool00000
0080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo
0=P0oooo0@0000010?ooo`001@00000;0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3o
ool0203oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0<0?ooo`030000003oool0oooo0=P0oooo0@0000010?ooo`0000@000000?ooo`3o
ool000002@3oool3000000@0oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool0
00000080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo0=P0oooo0@0000010?ooo`0000@0oooo0000003oool000002@3oool00`000000oooo0?ooo`03
0?ooo`050000003oool0oooo0?ooo`000000203oool01@000000oooo0?ooo`3oool000000080oooo
00<000000?ooo`3oool02P3oool2000000d0oooo00<000000?ooo`3oool0e`3oool200000040oooo
00020?ooo`8000002@3oool00`000000oooo0?ooo`0>0?ooo`<000000`3oool010000000oooo0?oo
o`00000<0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0f03oool100000040oooo
00030?ooo`030000003oool0oooo00L0oooo1000000?0?ooo`030000003oool0oooo0080oooo0P00
00080?ooo`<000000P3oool2000000d0oooo00<000000?ooo`3oool0f03oool100000040oooo000`
0?ooo`070000003oool0oooo0?ooo`000000oooo0000000=0?ooo`030000003oool0oooo0=L0oooo
0@0000010?ooo`00:P3oool3000000<0oooo00L000000?ooo`3oool0oooo0000003oool0000000d0
oooo00<000000?ooo`3oool09P3oool300000:h0oooo0@0000010?ooo`00<03oool010000000oooo
0?ooo`3oool3000000d0oooo00<000000?ooo`3oool0903oool2000000<0oooo0`00002Z0?ooo`80
00000@3oool002l0oooo0P0000030?ooo`030000003oool0000000d0oooo00<000000?ooo`3oool0
8P3oool2000000P0oooo0`00002X0?ooo`4000000@3oool00300oooo00@000000?ooo`3oool0oooo
0P00000>0?ooo`030000003oool0oooo0200oooo0P00000=0?ooo`800000YP3oool100000040oooo
000d0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool05P3oool:00000140oooo0P00
002T0?ooo`4000000@3oool003@0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0F
0?ooo`030000003oool0oooo01X0oooo0P00002R0?ooo`4000000@3oool003@0oooo00<000000?oo
o`3oool03@3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo01d0oooo0P00002P0?oo
o`4000000@3oool003@0oooo0P00000>0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3o
ool07`3oool2000009d0oooo0P0000010?ooo`00=03oool00`000000oooo0?ooo`0=0?ooo`030000
003oool0oooo01@0oooo00<000000?ooo`3oool08P3oool00`000000oooo0?ooo`2K0?ooo`400000
0@3oool003@0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0D0?ooo`030000003o
ool0oooo02<0oooo00<000000?ooo`3oool0VP3oool100000040oooo000d0?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`0U0?ooo`030000003oool0
oooo09T0oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo
01<0oooo00<000000?ooo`3oool09P3oool00`000000oooo0?ooo`2H0?ooo`4000000@3oool003@0
oooo0P00000>0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0:03oool00`000000
oooo0?ooo`2F0?ooo`8000000@3oool003@0oooo00<000000?ooo`3oool03@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0:@3oool00`000000oooo
0?ooo`2F0?ooo`4000000@3oool003@0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?oo
o`050?ooo`8000002`3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo09D0oooo0@00
00010?ooo`00=03oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`0000002@3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo09@0oooo0@000001
0?ooo`00=03oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00@0oooo00D000000?oo
o`3oool0oooo000000080?ooo`030000003oool0oooo02d0oooo00<000000?ooo`3oool0T`3oool1
00000040oooo000d0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0;`3oool00`00
0000oooo0?ooo`2B0?ooo`4000000@3oool003@0oooo0P00000>0?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0300
oooo00<000000?ooo`3oool0T03oool200000040oooo000d0?ooo`030000003oool0oooo00d0oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`050000003oool0oooo0?ooo`00
0000<`3oool00`000000oooo0?ooo`2A0?ooo`4000000@3oool000d0oooo0`0000030?ooo`050000
003oool0oooo0?ooo`0000001P3oool4000000<0oooo0P00000=0?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`060?ooo`040000003oool0oooo0000
03@0oooo00<000000?ooo`3oool0T03oool100000040oooo000@0?ooo`070000003oool0oooo0?oo
o`000000oooo000000090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?oo
o`070?ooo`800000=P3oool00`000000oooo0?ooo`2?0?ooo`4000000@3oool00100oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`080?ooo`050000003oool0oooo0?ooo`0000000P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`080?ooo`030000003oool0oooo03D0oooo00<000000?ooo`3oool0SP3o
ool100000040oooo000=0?ooo`<00000103oool00`000000oooo000000090?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`8000003P3oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo0400oooo00<000000?ooo`3oool0S@3oool200000040oooo000=
0?ooo`030000003oool0oooo00<0oooo00D000000?ooo`3oool0oooo000000080?ooo`050000003o
ool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00d0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`120?ooo`030000003oool0oooo08d0oooo
0@0000010?ooo`003@3oool00`000000oooo0?ooo`0>0?ooo`<000000`3oool010000000oooo0?oo
o`0000070?ooo`<000000P3oool2000000h0oooo00<000000?ooo`3oool00P3oool00`000000oooo
0?ooo`130?ooo`030000003oool0oooo08`0oooo0@0000010?ooo`003@3oool4000000l0oooo00<0
00000?ooo`3oool00P3oool2000000T0oooo00L000000?ooo`3oool0oooo0000003oool0000000d0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`140?ooo`030000003oool0oooo08/0
oooo0@0000010?ooo`00:P3oool3000000<0oooo00L000000?ooo`3oool0oooo0000003oool00000
00d0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`140?ooo`030000003oool0oooo
08/0oooo0@0000010?ooo`00<03oool010000000oooo0?ooo`3oool3000000h0oooo00@000000?oo
o`3oool00000B03oool00`000000oooo0?ooo`290?ooo`8000000@3oool002l0oooo0P0000030?oo
o`030000003oool0000000h0oooo00@000000?ooo`3oool00000B@3oool00`000000oooo0?ooo`29
0?ooo`4000000@3oool00300oooo00@000000?ooo`3oool0oooo0P00000?0?ooo`040000003oool0
oooo000004X0oooo00<000000?ooo`3oool0R03oool100000040oooo000d0?ooo`030000003oool0
oooo00h0oooo00@000000?ooo`3oool00000B`3oool00`000000oooo0?ooo`270?ooo`4000000@3o
ool003@0oooo00<000000?ooo`3oool03P3oool00`000000oooo0000001<0?ooo`030000003oool0
oooo08L0oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`0>0?ooo`030000003oool00000
04d0oooo00<000000?ooo`3oool0P@3oool00`000000oooo0?ooo`020?ooo`4000000@3oool003@0
oooo0P00000?0?ooo`030000003oool0000004h0oooo00<000000?ooo`3oool0O`3oool00`000000
oooo0?ooo`020?ooo`8000000@3oool003@0oooo00<000000?ooo`3oool03P3oool00`000000oooo
0000001?0?ooo`030000003oool0oooo07h0oooo00<000000?ooo`3oool00`3oool100000040oooo
000d0?ooo`030000003oool0oooo00h0oooo0P00001A0?ooo`030000003oool0oooo07`0oooo00<0
00000?ooo`3oool0103oool100000040oooo000d0?ooo`030000003oool0oooo00h0oooo0P00001B
0?ooo`030000003oool0oooo07/0oooo00<000000?ooo`3oool0103oool100000040oooo000d0?oo
o`030000003oool0oooo00h0oooo0P00001B0?ooo`030000003oool0oooo07X0oooo00<000000?oo
o`3oool01@3oool100000040oooo000d0?ooo`8000003`3oool2000005<0oooo00<000000?ooo`3o
ool0N@3oool00`000000oooo0?ooo`040?ooo`8000000@3oool003@0oooo00<000000?ooo`3oool0
3P3oool00`000000oooo0?ooo`1C0?ooo`030000003oool0oooo07L0oooo00<000000?ooo`3oool0
1P3oool100000040oooo000d0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0E03o
ool00`000000oooo0?ooo`1f0?ooo`030000003oool0oooo00H0oooo0@0000010?ooo`00=03oool0
0`000000oooo0?ooo`0>0?ooo`030000003oool0oooo05D0oooo00<000000?ooo`3oool0M03oool0
0`000000oooo0?ooo`070?ooo`4000000@3oool003@0oooo00<000000?ooo`3oool0I`3oool00`00
0000oooo0?ooo`1b0?ooo`030000003oool0oooo00P0oooo0@0000010?ooo`00=03oool00`000000
oooo0?ooo`1X0?ooo`800000LP3oool00`000000oooo0?ooo`080?ooo`4000000@3oool003@0oooo
0P00001[0?ooo`030000003oool0oooo06h0oooo00<000000?ooo`3oool0203oool200000040oooo
000d0?ooo`030000003oool0oooo06/0oooo00<000000?ooo`3oool0K@3oool00`000000oooo0?oo
o`090?ooo`4000000@3oool000040?ooo`0000000000000000@0oooo00<000000?ooo`3oool00P3o
ool3000000<0oooo00D000000?ooo`3oool0oooo000000060?ooo`@000000`3oool2000000d0oooo
00<000000?ooo`3oool0K03oool00`000000oooo0?ooo`1[0?ooo`030000003oool0oooo00X0oooo
0@0000010?ooo`00103oool00`000000oooo0?ooo`090?ooo`070000003oool0oooo0?ooo`000000
oooo000000090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo06d0oooo0P00000Y0?ooo`d00000=@3oool00`000000oooo0?ooo`0:0?oo
o`4000000@3oool000@0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo00P0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00X0oooo00<000000?ooo`3oool0K`3oool2000002<0oooo1000000=0?ooo`<00000<@3oool00`00
0000oooo0?ooo`0;0?ooo`4000000@3oool000040?ooo`0000000000000000T0oooo0`0000040?oo
o`030000003oool0000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0
oooo00X0oooo0P00001b0?ooo`<000007@3oool3000001@0oooo1000000]0?ooo`030000003oool0
oooo00X0oooo0P0000010?ooo`0000<0oooo0000003oool02P3oool00`000000oooo0?ooo`030?oo
o`050000003oool0oooo0?ooo`000000203oool01@000000oooo0?ooo`3oool000000080oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`1d0?ooo`800000603oool3000001/0oooo0`00
000Y0?ooo`030000003oool0oooo00`0oooo0@0000010?ooo`0000<0oooo0000003oool02P3oool0
0`000000oooo0?ooo`0>0?ooo`<000000`3oool010000000oooo0?ooo`0000070?ooo`<000000P3o
ool2000007L0oooo1000000A0?ooo`<000008@3oool3000002H0oooo00<000000?ooo`3oool0303o
ool100000040oooo00000`3oool0000000000002000000P0oooo1000000?0?ooo`030000003oool0
oooo0080oooo0P0000090?ooo`070000003oool0oooo0?ooo`000000oooo0000001j0?ooo`@00000
1`3oool6000002L0oooo0`00000R0?ooo`030000003oool0oooo00d0oooo0@0000010?ooo`00:P3o
ool3000000<0oooo00L000000?ooo`3oool0oooo0000003oool0000007h0oooo1`00000`0?ooo`<0
00007`3oool00`000000oooo0?ooo`0=0?ooo`4000000@3oool00300oooo00L000000?ooo`3oool0
oooo0000003oool000000;P0oooo0`00000L0?ooo`030000003oool0oooo00d0oooo0@0000010?oo
o`00;`3oool2000000<0oooo0`00002k0?ooo`<00000603oool00`000000oooo0?ooo`0=0?ooo`80
00000@3oool00300oooo00@000000?ooo`3oool0oooo0P00002o0?ooo`8000005P3oool00`000000
oooo0?ooo`0>0?ooo`4000000@3oool003@0oooo00<000000?ooo`3oool0`03oool2000001<0oooo
00<000000?ooo`3oool03`3oool100000040oooo000d0?ooo`030000003oool0oooo0<80oooo0P00
000A0?ooo`030000003oool0oooo00l0oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`34
0?ooo`8000003`3oool00`000000oooo0?ooo`0?0?ooo`4000000@3oool003@0oooo0P0000370?oo
o`800000303oool00`000000oooo0?ooo`0?0?ooo`8000000@3oool003@0oooo00<000000?ooo`3o
ool0b03oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0100oooo0@0000010?ooo`00
=03oool00`000000oooo0?ooo`390?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0
4@3oool100000040oooo000d0?ooo`030000003oool0oooo0<X0oooo0P0000070?ooo`030000003o
ool0oooo0140oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`3<0?ooo`030000003oool0
oooo00@0oooo00<000000?ooo`3oool04@3oool100000040oooo000d0?ooo`030000003oool0oooo
0<d0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0B0?ooo`4000000@3oool003@0
oooo0P00003?0?ooo`8000000P3oool00`000000oooo0?ooo`0A0?ooo`8000000@3oool003@0oooo
00<000000?ooo`3oool0d03oool2000001D0oooo0@0000010?ooo`00=03oool00`000000oooo0?oo
o`3A0?ooo`030000003oool0oooo01<0oooo0@0000010?ooo`00=03oool00`000000oooo0?ooo`3W
0?ooo`4000000@3oool003@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`090?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`090?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`090?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool00P3oool100000040oooo000d0?ooon/0
00000@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00><0
oooo00<000000?ooo`3oool0>P3oool00><0oooo0P00000k0?ooo`00F@3oool3000004L0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo00<0oooo00@0
00000?ooo`3oool0oooo0P00000k0?ooo`00G03oool00`000000oooo0?ooo`140?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`030?ooo`040000003o
ool0oooo0?ooo`<00000>P3oool003`0oooo0P0000030?ooo`@00000103oool2000000050?ooo`00
000000000000003oool00P0000020?ooo`800000103oool3000000<0oooo0P0000020?ooo`<00000
00<0oooo0000000000000P3oool2000000030?ooo`000000oooo00D00000203oool300000080oooo
100000020?ooo`@000000`3oool2000000<0oooo1000000:0?ooo`030000003oool0oooo00<0oooo
00@000000?ooo`3oool0oooo0`0000090?ooo`<00000503oool5000000@0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0?0?ooo`040000003oool0oooo0?ooo`<000002`3oool20000
0080oooo1P0000000`3oool000000?ooo`0200000080oooo0`0000090?ooo`00>`3oool010000000
oooo0?ooo`0000030?ooo`030000003oool0oooo00<0oooo00H000000?ooo`3oool000000?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000
oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool00000
0`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo00<000000?ooo`000000
1@3oool00`000000oooo0?ooo`070?ooo`040000003oool0oooo000000<0oooo00P000000?ooo`3o
ool000000?ooo`000000oooo00000080oooo00<000000?ooo`3oool02P3oool003/0oooo00@00000
0?ooo`3oool000000`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00@0
00000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000030?ooo`<00000103o
ool00`000000oooo0?ooo`070?ooo`<00000103oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo0080oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo00<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`090?ooo`030000
003oool0oooo00P0oooo100000090?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0000000<0oooo0P00000:0?ooo`030000
003oool0oooo00@0oooo00P000000?ooo`3oool000000?ooo`000000oooo00000080oooo0`00000:
0?ooo`00>`3oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool00P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo
00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
000000D0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`070?ooo`040000003oool0
oooo000000<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`020?ooo`040000003o
ool0oooo000000<0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`030?ooo`070000
003oool0oooo0?ooo`000000oooo000000090?ooo`030000003oool0000001H0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00d0oooo00L000000?oo
o`3oool0oooo0000003oool0000000`0oooo00<000000?ooo`3oool0103oool020000000oooo0?oo
o`000000oooo0000003oool000000P3oool010000000oooo0?ooo`0000090?ooo`00>`3oool01000
0000oooo0?ooo`0000020?ooo`030000003oool000000080000000<0oooo0000000000000P000002
0?ooo`<00000103oool2000000@0oooo0P000000103oool000000?ooo`3oool200000080oooo00@0
00000?ooo`0000000000103oool2000000D0oooo00<000000?ooo`3oool0203oool2000000<0oooo
00<000000?ooo`0000000P0000000`3oool000000?ooo`0300000080oooo0P0000030?ooo`030000
003oool00000008000002@3oool00`000000oooo0?ooo`030?ooo`040000003oool0oooo000000<0
oooo00<000000?ooo`3oool00`0000020?ooo`050000003oool0oooo0?ooo`0000005@3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0303oool20000
0080oooo0P0000020?ooo`<00000203oool4000000<0oooo0P0000020?ooo`80000000<0oooo0000
003oool00`3oool2000000X0oooo000l0?ooo`8000002P3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo02<0oooo00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo0080000000<0oooo0000000000000`3oool010000000oooo0?ooo`3oool20000
00030?ooo`000000000001<0oooo1@0000040?ooo`030000003oool0oooo00<0oooo00<000000?oo
o`3oool0803oool00`000000oooo0?ooo`0K0?ooo`00B03oool010000000oooo0?ooo`3oool20000
02@0oooo0P00000n0?ooo`050000003oool000000?ooo`000000B`3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo01@0oooo002b0?ooo`D00000J@3oool00?l0oooo8@3oool00?l0oooo
8@3oool00001\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {377.312, 200.375}} -> {-1.19034, \
6.65705, 0.00895744, 1.82235}}]
}, Open  ]],

Cell["\<\
This computes the solution using an orthogonal projection method \
with an explicit Runge-Kutta method used for the basic integration step. The \
initial step size and method order are the same as earlier, but the step size \
sequence in the integration may differ.\
\>", "Text"],

Cell[BoxData[
    \(solop = 
      NDSolve[eqs, vars, time, 
        Method \[Rule] {"\<OrthogonalProjection\>", \[IndentingNewLine]Method \
\[Rule] "\<ExplicitRungeKutta\>", \ 
            Dimensions \[Rule] Dimensions[Y]}]; \)], "Input",
  CellLabel->"In[18]:="],

Cell["\<\
Using the orthogonal projection method, the orthogonal error is \
reduced to approximately the level of roundoff in IEEE double-precision \
arithmetic.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(solop\  = \ 
      Y /. First[solop]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(OrthogonalErrorPlot[solop]; \)}], "Input",
  CellLabel->"In[19]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.47619 0.0147151 1.09952e+15 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.2619 -0.0125 -9 -9 ]
[.2619 -0.0125 9 0 ]
[.5 -0.0125 -3 -9 ]
[.5 -0.0125 3 0 ]
[.7381 -0.0125 -9 -9 ]
[.7381 -0.0125 9 0 ]
[.97619 -0.0125 -3 -9 ]
[.97619 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01472 -6 -4.5 ]
[-0.0125 .01472 0 4.5 ]
[-0.0125 .12467 -37.5 -6.25 ]
[-0.0125 .12467 0 6.25 ]
[-0.0125 .23462 -37.5 -6.25 ]
[-0.0125 .23462 0 6.25 ]
[-0.0125 .34457 -37.5 -6.25 ]
[-0.0125 .34457 0 6.25 ]
[-0.0125 .45452 -37.5 -6.25 ]
[-0.0125 .45452 0 6.25 ]
[-0.0125 .56447 -37.5 -6.25 ]
[-0.0125 .56447 0 6.25 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[.5 .63053 -113.125 0 ]
[.5 .63053 113.125 13.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.2619 0 m
.2619 .00625 L
s
[(0.5)] .2619 -0.0125 0 1 Mshowa
.5 0 m
.5 .00625 L
s
[(1)] .5 -0.0125 0 1 Mshowa
.7381 0 m
.7381 .00625 L
s
[(1.5)] .7381 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(2)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.07143 0 m
.07143 .00375 L
s
.11905 0 m
.11905 .00375 L
s
.16667 0 m
.16667 .00375 L
s
.21429 0 m
.21429 .00375 L
s
.30952 0 m
.30952 .00375 L
s
.35714 0 m
.35714 .00375 L
s
.40476 0 m
.40476 .00375 L
s
.45238 0 m
.45238 .00375 L
s
.54762 0 m
.54762 .00375 L
s
.59524 0 m
.59524 .00375 L
s
.64286 0 m
.64286 .00375 L
s
.69048 0 m
.69048 .00375 L
s
.78571 0 m
.78571 .00375 L
s
.83333 0 m
.83333 .00375 L
s
.88095 0 m
.88095 .00375 L
s
.92857 0 m
.92857 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01472 m
.00625 .01472 L
s
[(0)] -0.0125 .01472 1 0 Mshowa
0 .12467 m
.00625 .12467 L
s
gsave
-0.0125 .12467 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
0 .23462 m
.00625 .23462 L
s
gsave
-0.0125 .23462 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
0 .34457 m
.00625 .34457 L
s
gsave
-0.0125 .34457 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(3) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
0 .45452 m
.00625 .45452 L
s
gsave
-0.0125 .45452 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
0 .56447 m
.00625 .56447 L
s
gsave
-0.0125 .56447 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(5) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
.125 Mabswid
0 .03671 m
.00375 .03671 L
s
0 .0587 m
.00375 .0587 L
s
0 .08069 m
.00375 .08069 L
s
0 .10268 m
.00375 .10268 L
s
0 .14666 m
.00375 .14666 L
s
0 .16865 m
.00375 .16865 L
s
0 .19064 m
.00375 .19064 L
s
0 .21263 m
.00375 .21263 L
s
0 .25661 m
.00375 .25661 L
s
0 .2786 m
.00375 .2786 L
s
0 .30059 m
.00375 .30059 L
s
0 .32258 m
.00375 .32258 L
s
0 .36656 m
.00375 .36656 L
s
0 .38855 m
.00375 .38855 L
s
0 .41054 m
.00375 .41054 L
s
0 .43253 m
.00375 .43253 L
s
0 .47651 m
.00375 .47651 L
s
0 .4985 m
.00375 .4985 L
s
0 .52049 m
.00375 .52049 L
s
0 .54248 m
.00375 .54248 L
s
0 .58646 m
.00375 .58646 L
s
0 .60845 m
.00375 .60845 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.02381 .61178 m
.02381 .61803 L
s
.2619 .61178 m
.2619 .61803 L
s
.5 .61178 m
.5 .61803 L
s
.7381 .61178 m
.7381 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.07143 .61428 m
.07143 .61803 L
s
.11905 .61428 m
.11905 .61803 L
s
.16667 .61428 m
.16667 .61803 L
s
.21429 .61428 m
.21429 .61803 L
s
.30952 .61428 m
.30952 .61803 L
s
.35714 .61428 m
.35714 .61803 L
s
.40476 .61428 m
.40476 .61803 L
s
.45238 .61428 m
.45238 .61803 L
s
.54762 .61428 m
.54762 .61803 L
s
.59524 .61428 m
.59524 .61803 L
s
.64286 .61428 m
.64286 .61803 L
s
.69048 .61428 m
.69048 .61803 L
s
.78571 .61428 m
.78571 .61803 L
s
.83333 .61428 m
.83333 .61803 L
s
.88095 .61428 m
.88095 .61803 L
s
.92857 .61428 m
.92857 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
.99375 .12467 m
1 .12467 L
s
.99375 .23462 m
1 .23462 L
s
.99375 .34457 m
1 .34457 L
s
.99375 .45452 m
1 .45452 L
s
.99375 .56447 m
1 .56447 L
s
.125 Mabswid
.99625 .03671 m
1 .03671 L
s
.99625 .0587 m
1 .0587 L
s
.99625 .08069 m
1 .08069 L
s
.99625 .10268 m
1 .10268 L
s
.99625 .14666 m
1 .14666 L
s
.99625 .16865 m
1 .16865 L
s
.99625 .19064 m
1 .19064 L
s
.99625 .21263 m
1 .21263 L
s
.99625 .25661 m
1 .25661 L
s
.99625 .2786 m
1 .2786 L
s
.99625 .30059 m
1 .30059 L
s
.99625 .32258 m
1 .32258 L
s
.99625 .36656 m
1 .36656 L
s
.99625 .38855 m
1 .38855 L
s
.99625 .41054 m
1 .41054 L
s
.99625 .43253 m
1 .43253 L
s
.99625 .47651 m
1 .47651 L
s
.99625 .4985 m
1 .4985 L
s
.99625 .52049 m
1 .52049 L
s
.99625 .54248 m
1 .54248 L
s
.99625 .58646 m
1 .58646 L
s
.99625 .60845 m
1 .60845 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
gsave
.5 .63053 -174.125 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.5 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 12.812 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Orthogonal) show
129.000 12.812 moveto
(error) show
165.000 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
177.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
183.000 9.000 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(T) show
189.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
201.125 12.812 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
213.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(I) show
219.125 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
231.125 16.500 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(F) show
243.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(vs) show
261.250 12.812 moveto
(time) show
285.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .03975 m
.07281 .0309 L
.10075 .07599 L
.13955 .0255 L
.17377 .01472 L
.21124 .34055 L
.2479 .32891 L
.28579 .51987 L
.32382 .34055 L
.36256 .25886 L
.40174 .1762 L
.44103 .31138 L
.48007 .31702 L
.51893 .36538 L
.55751 .26212 L
.59584 .38121 L
.6339 .36243 L
.67197 .29322 L
.71036 .36683 L
.74904 .31759 L
.78772 .28076 L
.82661 .3434 L
.86567 .60332 L
.90499 .10103 L
.94059 .35179 L
.97619 .38346 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[19]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00<03oool200000340oooo0P0000040?ooo`030000003oool0oooo0080
oooo0`00000`0?ooo`@00000;`3oool4000000<0oooo00<000000?ooo`3oool00P3oool300000340
oooo100000080?ooo`00;`3oool010000000oooo0?ooo`00000_0?ooo`040000003oool0oooo0000
00/0oooo00<000000?ooo`3oool0;`3oool00`000000oooo0?ooo`0`0?ooo`030000003oool0oooo
00X0oooo00<000000?ooo`3oool0;P3oool00`000000oooo0?ooo`090?ooo`00;`3oool010000000
oooo0?ooo`00000_0?ooo`040000003oool0oooo000000/0oooo00<000000?ooo`3oool0;`3oool0
0`000000oooo0?ooo`0`0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0;`3oool0
0`000000oooo0?ooo`080?ooo`00;`3oool010000000oooo0?ooo`00000_0?ooo`040000003oool0
oooo000000P0oooo0`00000b0?ooo`030000003oool0oooo0300oooo00<000000?ooo`3oool01`3o
ool3000003<0oooo00<000000?ooo`3oool01`3oool002l0oooo00@000000?ooo`3oool00000;`3o
ool010000000oooo0?ooo`0000080?ooo`030000003oool0oooo0380oooo00<000000?ooo`3oool0
<03oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo03@0oooo00<000000?ooo`3oool0
1P3oool002l0oooo00@000000?ooo`3oool00000;`3oool010000000oooo0?ooo`0000080?ooo`03
0000003oool0oooo0300oooo0`00000`0?ooo`<000002@3oool00`000000oooo0?ooo`0a0?ooo`04
0000003oool0oooo000000P0oooo000`0?ooo`800000<@3oool2000000T0oooo1000000a0?ooo`03
0000003oool0oooo0300oooo00<000000?ooo`3oool01`3oool400000340oooo0P0000090?ooo`00
o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00:`3ooooa000000@0oooo000T0?ooo`80
00001@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo03H0oooo00<000000?ooo`3o
ool0=P3oool00`000000oooo0?ooo`0f0?ooo`030000003oool0oooo03L0oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`020?ooo`008`3oool010000000oooo0?ooo`0000040?ooo`03
0000003oool0oooo0>d0oooo00<000000?ooo`3oool00P3oool002<0oooo00@000000?ooo`3oool0
0000103oool3000002H0oooo0P0000340?ooo`800000103oool002<0oooo00@000000?ooo`3oool0
0000103oool00`000000oooo0?ooo`0S0?ooo`<0000000<0oooo0000003oool0a03oool00`000000
oooo0?ooo`020?ooo`008`3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0240
oooo0P0000040?ooo`030000003oool0oooo0<<0oooo00<000000?ooo`3oool00P3oool002<0oooo
00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`0O0?ooo`8000001P3oool00`000000
oooo0?ooo`330?ooo`030000003oool0oooo0080oooo000T0?ooo`8000001@3oool00`000000oooo
0?ooo`0<0?ooo`@000003P3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo0<<0oooo
00<000000?ooo`3oool00P3oool002/0oooo0P0000070?ooo`H00000103oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0`@3oool2000000@0oooo000[
0?ooo`030000003oool0oooo00<0oooo0`00000:0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`320?ooo`030000003oool0oooo0080oooo000[0?oo
o`030000003oool0oooo0140oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`090?oo
o`030000003oool0oooo0<80oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3o
ool04@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3o
ool0`P3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0B0?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`320?ooo`030000
003oool0oooo0080oooo000[0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0<80oooo00<000000?ooo`3oool00P3o
ool002/0oooo0P00000D0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool03@3oool0
0`000000oooo0?ooo`310?ooo`800000103oool002/0oooo00<000000?ooo`3oool04`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool0`P3oool00`00
0000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0D0?ooo`040000003oool0oooo0000
0100oooo00<000000?ooo`3oool0`P3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000
oooo0?ooo`0D0?ooo`030000003oool000000180oooo00<000000?ooo`3oool0`@3oool00`000000
oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0140oooo
00<000000?ooo`3oool0`@3oool00`000000oooo0?ooo`020?ooo`00:`3oool2000002X0oooo00<0
00000?ooo`3oool0`03oool2000000@0oooo000[0?ooo`030000003oool0oooo02T0oooo00<00000
0?ooo`3oool0`@3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0Y0?oo
o`030000003oool0oooo0<40oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3o
ool0:@3oool00`000000oooo0?ooo`310?ooo`030000003oool0oooo0080oooo000[0?ooo`030000
003oool0oooo02T0oooo00<000000?ooo`3oool0`@3oool00`000000oooo0?ooo`020?ooo`00:`3o
ool2000002X0oooo00<000000?ooo`3oool0ZP3oool00`000000oooo0?ooo`0C0?ooo`800000103o
ool002/0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`2Y0?ooo`030000003oool0
oooo01@0oooo00<000000?ooo`3oool00P3oool000@0oooo100000020?ooo`050000003oool0oooo
0?ooo`0000001P3oool4000000<0oooo0P00000=0?ooo`030000003oool0oooo02X0oooo00<00000
0?ooo`3oool0Z@3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo0080oooo00060?oo
o`030000003oool0oooo0080oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool0
00000080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0
oooo0:T0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`020?ooo`001P3oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo00P0oooo00D000000?ooo`3oool0oooo00000002
0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`2Y
0?ooo`8000005@3oool00`000000oooo0?ooo`020?ooo`001P3oool00`000000oooo0?ooo`020?oo
o`030000003oool0000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0
oooo00X0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`2Y0?ooo`8000005@3oool0
0`000000oooo0?ooo`020?ooo`001P3oool01@000000oooo0?ooo`3oool0000000<0oooo00<00000
0?ooo`3oool01P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3o
ool3000002X0oooo00<000000?ooo`3oool0Z@3oool2000001@0oooo0P0000040?ooo`00103oool3
000000h0oooo0`0000030?ooo`040000003oool0oooo000000`0oooo00<000000?ooo`3oool0:P3o
ool00`000000oooo0?ooo`2X0?ooo`030000003oool0000001D0oooo00<000000?ooo`3oool00P3o
ool000H0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`020?ooo`800000203oool3
00000080oooo0P00000[0?ooo`030000003oool0oooo0:P0oooo00<000000?ooo`0000005@3oool0
0`000000oooo0?ooo`020?ooo`009`3oool01`000000oooo0?ooo`3oool000000?ooo`000000:P3o
ool00`000000oooo0?ooo`2X0?ooo`030000003oool0000001D0oooo00<000000?ooo`3oool00P3o
ool00240oooo0`0000030?ooo`040000003oool0oooo0?ooo`<00000:`3oool00`000000oooo0?oo
o`2W0?ooo`030000003oool0000001D0oooo00<000000?ooo`3oool00P3oool002L0oooo00@00000
0?ooo`3oool0oooo0P00000/0?ooo`030000003oool0oooo0:L0oooo00@000000?ooo`3oool00000
4`3oool2000000@0oooo000V0?ooo`8000000`3oool00`000000oooo0?ooo`0[0?ooo`030000003o
ool0oooo0:L0oooo00@000000?ooo`3oool00000503oool00`000000oooo0?ooo`020?ooo`009`3o
ool010000000oooo0?ooo`3oool3000002/0oooo00<000000?ooo`3oool0Y`3oool010000000oooo
0?ooo`00000D0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo02/0oooo
00<000000?ooo`3oool0Y`3oool010000000oooo0?ooo`00000D0?ooo`030000003oool0oooo0080
oooo000[0?ooo`030000003oool0oooo02/0oooo00<000000?ooo`3oool0Y`3oool010000000oooo
0?ooo`00000D0?ooo`030000003oool0oooo0080oooo000[0?ooo`800000;03oool00`000000oooo
0?ooo`2W0?ooo`040000003oool0oooo000001<0oooo0P0000040?ooo`00:`3oool00`000000oooo
0?ooo`0[0?ooo`030000003oool0oooo0:L0oooo00D000000?ooo`3oool0oooo0000000C0?ooo`03
0000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo02/0oooo00<000000?ooo`3oool0
<03oool00`000000oooo0?ooo`1d0?ooo`050000003oool0oooo0?ooo`0000004`3oool00`000000
oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo02l0oooo
00<000000?ooo`3oool0M03oool01@000000oooo0?ooo`3oool0000001<0oooo00<000000?ooo`3o
ool00P3oool002/0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0^0?ooo`030000
003oool0000007D0oooo00D000000?ooo`3oool0oooo0000000C0?ooo`030000003oool0oooo0080
oooo000[0?ooo`800000;@3oool00`000000oooo0?ooo`0^0?ooo`030000003oool0000007@0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0@0?ooo`800000103oool002/0oooo00<0
00000?ooo`3oool0;03oool00`000000oooo0?ooo`0]0?ooo`040000003oool0oooo000007@0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo0080oooo
000[0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool0;@3oool010000000oooo0?oo
o`00001d0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool04@3oool00`000000oooo
0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo02`0oooo00<0
00000?ooo`3oool00P3oool00`000000oooo0?ooo`1a0?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool0403oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0/
0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`1a
0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`02
0?ooo`00:`3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo02/0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`1a0?ooo`030000003oool0oooo00<0oooo00<000000?oo
o`3oool0403oool00`000000oooo0?ooo`020?ooo`00:`3oool2000002h0oooo00<000000?ooo`3o
ool0:P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0700oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0?0?ooo`800000103oool002/0oooo00<000000?ooo`3oool0
;@3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
L03oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0
0P3oool000@0oooo100000020?ooo`050000003oool0oooo0?ooo`0000001P3oool4000000<0oooo
0P00000=0?ooo`030000003oool0oooo02d0oooo00<000000?ooo`3oool0:@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo0700oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`0@0?ooo`030000003oool0oooo0080oooo00040?ooo`030000003oool0oooo00@0oooo00<0
00000?ooo`0000002@3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0
2P3oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo02P0oooo00<000000?ooo`3oool0
1P3oool00`000000oooo0?ooo`1`0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
403oool00`000000oooo0?ooo`020?ooo`001@3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo00P0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo
00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`0X0?ooo`030000003oool0oooo00L0oooo
00<000000?ooo`3oool0K`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00l0oooo
00<000000?ooo`3oool00P3oool000H0oooo00<000000?ooo`3oool00P3oool00`000000oooo0000
00090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`<00000
;@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0
K`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00h0oooo0P0000040?ooo`001`3o
ool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0;@3oool00`000000
oooo0?ooo`0W0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0K`3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool00P3oool000@0oooo
00@000000?ooo`3oool000003@3oool3000000<0oooo00@000000?ooo`3oool000001`3oool30000
0080oooo0P00000^0?ooo`030000003oool0oooo02H0oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`1^0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`020?ooo`001@3oool200000100oooo00<000000?ooo`3oool00P3oool2000000T0
oooo00L000000?ooo`3oool0oooo0000003oool0000002d0oooo00<000000?ooo`3oool09P3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo06d0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`0?0?ooo`030000003oool0oooo0080oooo000Q0?ooo`<000000`3oool01000
0000oooo0?ooo`3oool3000002h0oooo00<000000?ooo`3oool0903oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo06d0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0?
0?ooo`030000003oool0oooo0080oooo000W0?ooo`040000003oool0oooo0?ooo`800000;`3oool0
0`000000oooo0?ooo`0T0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0K@3oool0
0`000000oooo0?ooo`060?ooo`030000003oool0oooo00d0oooo0P0000040?ooo`009P3oool20000
00<0oooo00<000000?ooo`3oool0;P3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo
00d0oooo00<000000?ooo`3oool0K03oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo
00h0oooo00<000000?ooo`3oool00P3oool002L0oooo00@000000?ooo`3oool0oooo0`00000^0?oo
o`030000003oool0oooo02<0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0L0?oo
o`030000003oool0oooo04d0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0>0?oo
o`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3o
ool08P3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3o
ool0C@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3o
ool00P3oool002/0oooo00<000000?ooo`3oool0;P3oool00`000000oooo0?ooo`0R0?ooo`030000
003oool0oooo00h0oooo00<000000?ooo`3oool06`3oool00`000000oooo0000001>0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`020?ooo`00:`3o
ool2000002l0oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`0@0?ooo`030000003o
ool0oooo01X0oooo00<000000?ooo`000000CP3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo00d0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`0^0?ooo`030000003oool0
oooo0240oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0J0?ooo`030000003oool0
000003D0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`060?ooo`030000003oool0
oooo00h0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0;P3oool00`00
0000oooo0?ooo`0P0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool06@3oool01@00
0000oooo0?ooo`3oool0000003<0oooo00<000000?ooo`0000005`3oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?oo
o`3oool0;`3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo0180oooo00<000000?oo
o`3oool0603oool01@000000oooo0?ooo`3oool000000380oooo00@000000?ooo`3oool000005`3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool00P3o
ool002/0oooo00<000000?ooo`3oool0;`3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0
oooo01<0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo01H0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo0080oooo000[0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool07P3oool00`00
0000oooo0?ooo`0C0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`0D0?ooo`030000003oool0000001H0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`0B0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool03@3oool00`00
0000oooo0?ooo`020?ooo`00:`3oool200000300oooo00<000000?ooo`3oool07@3oool00`000000
oooo0?ooo`0D0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`0D0?ooo`030000003oool0000001D0oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`0B0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0303oool2000000@0
oooo000[0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool07@3oool00`000000oooo
0?ooo`0E0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`0C0?ooo`050000003oool0oooo0?ooo`0000004`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo0140oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool0
703oool00`000000oooo0?ooo`0F0?ooo`<000005@3oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo0180oooo00D000000?ooo`3oool0oooo0000000B0?ooo`030000003oool0oooo00T0
oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0;`3oool00`000000oooo
0?ooo`0L0?ooo`030000003oool0oooo01T0oooo1000000A0?ooo`030000003oool0oooo00D0oooo
00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo
00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00T0oooo
00<000000?ooo`3oool0303oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?oo
o`0_0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool07P3oool3000000d0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool03@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00l0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0080oooo000[
0?ooo`800000<@3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo0240oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo0100oooo00<00000
0?ooo`3oool01@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00h0oooo00<00000
0?ooo`3oool03P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00/0oooo0P000004
0?ooo`00:`3oool00`000000oooo0?ooo`0`0?ooo`030000003oool0oooo01T0oooo00<000000?oo
o`3oool08`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool03`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00/0oooo00<000000?oo
o`3oool03P3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00T0oooo00<000000?oo
o`3oool0303oool00`000000oooo0?ooo`020?ooo`001@3oool2000000<0oooo00D000000?ooo`3o
ool0oooo000000060?ooo`@000000`3oool2000000d0oooo00<000000?ooo`3oool0<03oool00`00
0000oooo0?ooo`040?ooo`<000004P3oool00`000000oooo0?ooo`0T0?ooo`030000003oool0oooo
00L0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo
00L0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo
00d0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo
0080oooo00040?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`0000002@3oool01@00
0000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0`
0?ooo`030000003oool0oooo00@000000P3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0
oooo02H0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`080?ooo`030000003oool0
oooo00h0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`070?ooo`030000003oool0
oooo0180oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool00P3oool000L0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0oooo0300oooo0`0000060?ooo`030000003oool0oooo00l0oooo00<00000
0?ooo`3oool09`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool0303oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00H0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00X0oooo00<00000
0?ooo`3oool02`3oool00`000000oooo0?ooo`020?ooo`001@3oool2000000@0oooo00<000000?oo
o`0000002@3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool3
000003X0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0
oooo00@0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo00X0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0
oooo00/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?ooo`800000103oool0
00L0oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`060?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo03X0oooo00<0
00000?ooo`3oool03P3oool00`000000oooo0?ooo`0X0?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00/0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00/0oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0080oooo0004
0?ooo`040000003oool0oooo000000d0oooo0`0000030?ooo`040000003oool0oooo000000L0oooo
0`0000020?ooo`800000>`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo02X0oooo
00D000000?ooo`3oool0oooo0000000>0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3o
ool0303oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool00P3oool000D0oooo0P00000@0?ooo`030000003oool0oooo0080oooo0P0000090?ooo`070000
003oool0oooo0?ooo`000000oooo0000000j0?ooo`030000003oool0oooo00d0oooo00<000000?oo
o`3oool0:`3oool010000000oooo0?ooo`00000>0?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool03@3oool01@000000oooo0?ooo`3oool0000001T0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool00P3o
ool00240oooo0`0000030?ooo`040000003oool0oooo0?ooo`<00000>P3oool00`000000oooo0?oo
o`0=0?ooo`030000003oool0oooo02`0oooo0P00000?0?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool03P3oool010000000oooo0?ooo`00000J0?ooo`030000003oool0oooo00T0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0080oooo
000W0?ooo`050000003oool0oooo0?ooo`000000?@3oool00`000000oooo0?ooo`0<0?ooo`030000
003oool0oooo02d0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`060?ooo`800000
4@3oool00`000000oooo0000000K0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
3P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo0080oooo000V0?ooo`8000000`3o
ool2000003`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0n0?ooo`030000003o
ool0oooo00@0oooo0P00000D0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool02@3o
ool00`000000oooo0?ooo`0?0?ooo`8000001`3oool2000000@0oooo000W0?ooo`040000003oool0
oooo0?ooo`<00000>`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo03l0oooo00<0
00000?ooo`3oool00P3oool2000003<0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?oo
o`0A0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00P3oool002/0oooo00<00000
0?ooo`3oool0>`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0400oooo00@00000
0?ooo`0000000000=@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0180oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0k
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0@03oool2000003L0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0?03oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo07T0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0D0?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool002/0oooo0P00000m0?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool0N`3oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo01P0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`0l0?ooo`030000
003oool0oooo00T0oooo00<000000?ooo`3oool0N`3oool00`000000oooo0?ooo`080?ooo`030000
003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0?03o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo07/0oooo00<000000?ooo`3oool0203o
ool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0
oooo03`0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`1k0?ooo`030000003oool0
oooo00P0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`00
0000oooo0?ooo`0m0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0N`3oool00`00
0000oooo0?ooo`080?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool002/0
oooo0P00000n0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0O03oool00`000000
oooo0?ooo`070?ooo`030000003oool0oooo01T0oooo0P0000040?ooo`00:`3oool00`000000oooo
0?ooo`0m0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0O03oool00`000000oooo
0?ooo`070?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool00P3oool002/0oooo00<0
00000?ooo`3oool0?@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo07d0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo0080oooo000[
0?ooo`030000003oool0oooo03d0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`1m
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`02
0?ooo`00:`3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool0O@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01X0oooo00<000000?oo
o`3oool00P3oool002/0oooo0P00000o0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool0OP3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01T0oooo0P0000040?ooo`00
:`3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
OP3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0
0P3oool000H0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0oooo00@0oooo
100000030?ooo`8000003@3oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo00D0oooo
00<000000?ooo`3oool0OP3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01X0oooo
00<000000?ooo`3oool00P3oool000H0oooo00<000000?ooo`3oool00P3oool00`000000oooo0000
00090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo03h0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`1n0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`020?ooo`000`3o
ool5000000@0oooo00<000000?ooo`3oool0203oool01@000000oooo0?ooo`3oool000000080oooo
00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0o0?ooo`030000003oool0oooo00@0oooo
00<000000?ooo`3oool0O`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01X0oooo
00<000000?ooo`3oool00P3oool000<0oooo00@000000?ooo`3oool00000103oool00`000000oooo
000000090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`03
0000003oool0oooo03l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`200?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`020?ooo`00
103oool00`000000oooo000000030?ooo`050000003oool0oooo0?ooo`000000203oool01@000000
oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool3000003l0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`200?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool06@3oool2000000@0oooo00050?ooo`8000003P3oool3000000<0oooo00@000000?ooo`3o
ool00000303oool00`000000oooo0?ooo`0o0?ooo`030000003oool0oooo00<0oooo00<000000?oo
o`3oool0P03oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01X0oooo00<000000?oo
o`3oool00P3oool000H0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`020?ooo`80
0000203oool300000080oooo0P0000100?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool0P03oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3o
ool00P3oool002L0oooo00L000000?ooo`3oool0oooo0000003oool000000400oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`200?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool06`3oool00`000000oooo0?ooo`020?ooo`008@3oool3000000<0oooo00@000000?ooo`3o
ool0oooo0`0000100?ooo`050000003oool0oooo0?ooo`000000P`3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool00P3oool002L0oooo00@000000?oo
o`3oool0oooo0P0000110?ooo`050000003oool0oooo0?ooo`000000Q03oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo01X0oooo0P0000040?ooo`009P3oool2000000<0oooo00<00000
0?ooo`3oool0@03oool01@000000oooo0?ooo`3oool0000008@0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo0080oooo000W0?ooo`040000003oool0
oooo0?ooo`<00000@03oool01@000000oooo0?ooo`3oool0000008@0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003o
ool0oooo0440oooo00<000000?ooo`000000Q@3oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo01/0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0@@3oool0
0`000000oooo000000250?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool06`3oool0
0`000000oooo0?ooo`020?ooo`00:`3oool200000480oooo00<000000?ooo`000000Q@3oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo01X0oooo0P0000040?ooo`00:`3oool00`000000
oooo0?ooo`110?ooo`030000003oool0000008H0oooo00<000000?ooo`3oool00P3oool00`000000
oooo0?ooo`0K0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo0440oooo
00<000000?ooo`000000QP3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01/0oooo
00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0@P3oool00`000000oooo0?oo
o`250?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool06`3oool00`000000oooo0?oo
o`020?ooo`00:`3oool00`000000oooo0?ooo`120?ooo`030000003oool0oooo08D0oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo0080oooo000[0?oo
o`030000003oool0oooo0480oooo00<000000?ooo`3oool0Q@3oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo01/0oooo00<000000?ooo`3oool00P3oool002/0oooo0P00003;0?ooo`05
0000003oool0oooo0?ooo`0000007@3oool2000000@0oooo000[0?ooo`030000003oool0oooo0<X0
oooo00D000000?ooo`3oool0oooo0000000N0?ooo`030000003oool0oooo0080oooo000[0?ooo`03
0000003oool0oooo0</0oooo00@000000?ooo`3oool000007P3oool00`000000oooo0?ooo`020?oo
o`00:`3oool00`000000oooo0?ooo`3;0?ooo`040000003oool0oooo000001h0oooo00<000000?oo
o`3oool00P3oool002/0oooo00<000000?ooo`3oool0b`3oool010000000oooo0?ooo`00000N0?oo
o`030000003oool0oooo0080oooo000[0?ooo`800000c03oool010000000oooo0?ooo`00000M0?oo
o`800000103oool002/0oooo00<000000?ooo`3oool0b`3oool010000000oooo0?ooo`00000N0?oo
o`030000003oool0oooo0080oooo00040?ooo`<000000`3oool01@000000oooo0?ooo`3oool00000
00H0oooo100000030?ooo`8000003@3oool00`000000oooo0?ooo`3;0?ooo`040000003oool0oooo
000001h0oooo00<000000?ooo`3oool00P3oool000L0oooo00L000000?ooo`3oool0oooo0000003o
ool0000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo
00<000000?ooo`3oool0b`3oool010000000oooo0?ooo`00000N0?ooo`030000003oool0oooo0080
oooo00070?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0203oool01@000000oooo
0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`3<0?ooo`03
0000003oool0000001h0oooo00<000000?ooo`3oool00P3oool000@0oooo0`0000040?ooo`030000
003oool0000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0
oooo0`00003<0?ooo`030000003oool0000001d0oooo0P0000040?ooo`00103oool00`000000oooo
0?ooo`030?ooo`050000003oool0oooo0?ooo`000000203oool01@000000oooo0?ooo`3oool00000
0080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`3<0?ooo`030000003oool00000
01h0oooo00<000000?ooo`3oool00P3oool000@0oooo00<000000?ooo`3oool03P3oool3000000<0
oooo00@000000?ooo`3oool000001`3oool300000080oooo0P00003=0?ooo`030000003oool00000
01h0oooo00<000000?ooo`3oool00P3oool000@0oooo1000000?0?ooo`030000003oool0oooo0080
oooo0P0000090?ooo`070000003oool0oooo0?ooo`000000oooo0000003<0?ooo`8000007`3oool0
0`000000oooo0?ooo`020?ooo`008@3oool3000000<0oooo00@000000?ooo`3oool0oooo0`00003<
0?ooo`8000007`3oool00`000000oooo0?ooo`020?ooo`009`3oool010000000oooo0?ooo`3oool2
00000<d0oooo0P00000N0?ooo`800000103oool002H0oooo0P0000030?ooo`030000003oool0oooo
0<d0oooo00<000000?ooo`3oool07@3oool00`000000oooo0?ooo`020?ooo`009`3oool010000000
oooo0?ooo`3oool300000<d0oooo00<000000?ooo`3oool07@3oool00`000000oooo0?ooo`020?oo
o`00:`3oool00`000000oooo0?ooo`3=0?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3o
ool00P3oool002/0oooo00<000000?ooo`3oool0c@3oool00`000000oooo0?ooo`0M0?ooo`030000
003oool0oooo0080oooo000[0?ooo`030000003oool0oooo0>d0oooo00<000000?ooo`3oool00P3o
ool002/0oooo0P0000040?ooo`030000003oool0oooo03H0oooo00<000000?ooo`3oool0=P3oool0
0`000000oooo0?ooo`0f0?ooo`030000003oool0oooo03L0oooo00@000000?ooo`3oool0oooo0P00
00040?ooo`00:`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool0203oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo000[0?oo
oo400000103oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool0
0=d0oooo00<000000?ooo`3oool0@03oool00=d0oooo0P0000110?ooo`00D`3oool3000004L0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo00<0oooo
00@000000?ooo`3oool0oooo0P0000110?ooo`00EP3oool00`000000oooo0?ooo`140?ooo`030000
003oool0oooo00<0oooo00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`030?ooo`040000
003oool0oooo0?ooo`<00000@03oool003H0oooo0P0000030?ooo`@00000103oool2000000050?oo
o`00000000000000003oool00P0000020?ooo`800000103oool3000000<0oooo0P0000020?ooo`<0
000000<0oooo0000000000000P3oool2000000030?ooo`000000oooo00D00000203oool300000080
oooo100000020?ooo`@000000`3oool2000000<0oooo1000000:0?ooo`030000003oool0oooo00<0
oooo00@000000?ooo`3oool0oooo0`0000090?ooo`<00000503oool5000000@0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`0?0?ooo`040000003oool0oooo0?ooo`<000002`3oool2
00000080oooo1P0000000`3oool000000?ooo`0200000080oooo0`00000?0?ooo`00=@3oool01000
0000oooo0?ooo`0000030?ooo`030000003oool0oooo00<0oooo00H000000?ooo`3oool000000?oo
o`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool01000
0000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000
0P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool0
00000`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo00<000000?ooo`00
00001@3oool00`000000oooo0?ooo`070?ooo`040000003oool0oooo000000<0oooo00P000000?oo
o`3oool000000?ooo`000000oooo00000080oooo00<000000?ooo`3oool0403oool003D0oooo00@0
00000?ooo`3oool000000`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo
00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000030?ooo`<00000
103oool00`000000oooo0?ooo`070?ooo`<00000103oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo0080oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`0:0?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo00P0oooo100000090?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0000000<0oooo0P00000:0?ooo`03
0000003oool0oooo00@0oooo00P000000?ooo`3oool000000?ooo`000000oooo00000080oooo0`00
000@0?ooo`00=@3oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool00P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080
oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0
oooo000000D0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`070?ooo`040000003o
ool0oooo000000<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`020?ooo`040000
003oool0oooo000000<0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`030?ooo`07
0000003oool0oooo0?ooo`000000oooo000000030?ooo`<000000`3oool00`000000oooo0000000F
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0=
0?ooo`070000003oool0oooo0?ooo`000000oooo0000000<0?ooo`030000003oool0oooo00@0oooo
00P000000?ooo`3oool000000?ooo`000000oooo00000080oooo00@000000?ooo`3oool000003`3o
ool003D0oooo00@000000?ooo`3oool000000P3oool00`000000oooo00000002000000030?ooo`00
00000000008000000P3oool3000000@0oooo0P0000040?ooo`80000000@0oooo0000003oool0oooo
0P0000020?ooo`040000003oool00000000000@0oooo0P0000050?ooo`030000003oool0oooo00P0
oooo0P0000030?ooo`030000003oool000000080000000<0oooo0000003oool00`0000020?ooo`80
00000`3oool00`000000oooo00000002000000T0oooo00<000000?ooo`3oool00`3oool010000000
oooo0?ooo`0000030?ooo`050000003oool0oooo0?ooo`0000000`3oool01@000000oooo0?ooo`3o
ool0000001D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo00`0oooo0P0000020?ooo`8000000P3oool3000000P0oooo100000030?ooo`8000000P3o
ool2000000030?ooo`000000oooo00<0oooo0P00000@0?ooo`00=P3oool2000000X0oooo00<00000
0?ooo`3oool00P3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo02d0oooo00<00000
0?ooo`3oool00`3oool00`000000oooo0?ooo`020000000:0?ooo`00000000000?ooo`000000oooo
0000003oool000000?ooo`80000000<0oooo0000000000004`3oool5000000@0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`0P0?ooo`030000003oool0oooo0240oooo00120?ooo`04
0000003oool0oooo0?ooo`800000903oool2000003h0oooo1@00001;0?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool06P3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {276.312, 99.375}} -> {-1.08596, \
3.17483, 0.00874385, 3.78686}}]
}, Open  ]],

Cell["\<\
The Schulz iteration, using the incremental formulation, generally \
yields smaller errors than the direct singular value decomposition.\
\>", \
"Text"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.354926 0.00413632 0.0147151 0.00443884 [
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
0 g
.25 Mabswid
[ ] 0 Mabsdash
.25537 .11267 m
.25537 .11762 L
s
[(0)] .24141 .07724 -1 -1 Mshowa
.43117 .11267 m
.43117 .11762 L
s
[(0.5)] .39239 .07724 -1 -1 Mshowa
.60699 .11267 m
.60699 .11762 L
s
[(1)] .59122 .07724 -1 -1 Mshowa
.7828 .11267 m
.7828 .11762 L
s
[(1.5)] .74312 .07724 -1 -1 Mshowa
.95861 .11267 m
.95861 .11762 L
s
[(2)] .94465 .07724 -1 -1 Mshowa
.125 Mabswid
.29053 .11267 m
.29053 .11564 L
s
.32569 .11267 m
.32569 .11564 L
s
.36086 .11267 m
.36086 .11564 L
s
.39602 .11267 m
.39602 .11564 L
s
.46634 .11267 m
.46634 .11564 L
s
.5015 .11267 m
.5015 .11564 L
s
.53666 .11267 m
.53666 .11564 L
s
.57182 .11267 m
.57182 .11564 L
s
.64215 .11267 m
.64215 .11564 L
s
.67731 .11267 m
.67731 .11564 L
s
.71248 .11267 m
.71248 .11564 L
s
.74764 .11267 m
.74764 .11564 L
s
.81796 .11267 m
.81796 .11564 L
s
.85312 .11267 m
.85312 .11564 L
s
.88828 .11267 m
.88828 .11564 L
s
.92344 .11267 m
.92344 .11564 L
s
.25 Mabswid
.23779 .11267 m
.97619 .11267 L
s
.23779 .12433 m
.2424 .12433 L
s
[(0)] .20684 .11185 -1 -1 Mshowa
.23779 .20342 m
.2424 .20342 L
s
[(2)] .07344 .18456 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .18456 -1 -1 Mshowa
P
[(10)] .12308 .18456 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .20786 -1 -1 Mshowa
P
.23779 .28251 m
.2424 .28251 L
s
[(4)] .07344 .26365 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .26365 -1 -1 Mshowa
P
[(10)] .12308 .26365 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .28695 -1 -1 Mshowa
P
.23779 .3616 m
.2424 .3616 L
s
[(6)] .07344 .34274 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .34274 -1 -1 Mshowa
P
[(10)] .12308 .34274 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .36604 -1 -1 Mshowa
P
.23779 .4407 m
.2424 .4407 L
s
[(8)] .07344 .42183 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .42183 -1 -1 Mshowa
P
[(10)] .12308 .42183 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-16)] .17272 .44514 -1 -1 Mshowa
P
.23779 .51979 m
.2424 .51979 L
s
[(1)] .07344 .50092 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .50092 -1 -1 Mshowa
P
[(10)] .12308 .50092 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-15)] .17272 .52423 -1 -1 Mshowa
P
.23779 .59888 m
.2424 .59888 L
s
[(1.2)] .02381 .58001 -1 -1 Mshowa
p
%%IncludeResource: font Math1Mono
%%IncludeFont: Math1Mono
/Math1Mono findfont 10 scalefont setfont
[(\\264)] .09826 .58001 -1 -1 Mshowa
P
[(10)] .12308 .58001 -1 -1 Mshowa
p
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 7.5 scalefont setfont
[(-15)] .17272 .60332 -1 -1 Mshowa
P
.125 Mabswid
.23779 .1441 m
.24055 .1441 L
s
.23779 .16387 m
.24055 .16387 L
s
.23779 .18364 m
.24055 .18364 L
s
.23779 .22319 m
.24055 .22319 L
s
.23779 .24296 m
.24055 .24296 L
s
.23779 .26274 m
.24055 .26274 L
s
.23779 .30228 m
.24055 .30228 L
s
.23779 .32206 m
.24055 .32206 L
s
.23779 .34183 m
.24055 .34183 L
s
.23779 .38138 m
.24055 .38138 L
s
.23779 .40115 m
.24055 .40115 L
s
.23779 .42092 m
.24055 .42092 L
s
.23779 .46047 m
.24055 .46047 L
s
.23779 .48024 m
.24055 .48024 L
s
.23779 .50002 m
.24055 .50002 L
s
.23779 .53956 m
.24055 .53956 L
s
.23779 .55934 m
.24055 .55934 L
s
.23779 .57911 m
.24055 .57911 L
s
.25 Mabswid
.23779 .11267 m
.23779 .6024 L
s
.25537 .59745 m
.25537 .6024 L
s
.43117 .59745 m
.43117 .6024 L
s
.60699 .59745 m
.60699 .6024 L
s
.7828 .59745 m
.7828 .6024 L
s
.95861 .59745 m
.95861 .6024 L
s
.125 Mabswid
.29053 .59943 m
.29053 .6024 L
s
.32569 .59943 m
.32569 .6024 L
s
.36086 .59943 m
.36086 .6024 L
s
.39602 .59943 m
.39602 .6024 L
s
.46634 .59943 m
.46634 .6024 L
s
.5015 .59943 m
.5015 .6024 L
s
.53666 .59943 m
.53666 .6024 L
s
.57182 .59943 m
.57182 .6024 L
s
.64215 .59943 m
.64215 .6024 L
s
.67731 .59943 m
.67731 .6024 L
s
.71248 .59943 m
.71248 .6024 L
s
.74764 .59943 m
.74764 .6024 L
s
.81796 .59943 m
.81796 .6024 L
s
.85312 .59943 m
.85312 .6024 L
s
.88828 .59943 m
.88828 .6024 L
s
.92344 .59943 m
.92344 .6024 L
s
.25 Mabswid
.23779 .6024 m
.97619 .6024 L
s
.97157 .12433 m
.97619 .12433 L
s
.97157 .20342 m
.97619 .20342 L
s
.97157 .28251 m
.97619 .28251 L
s
.97157 .3616 m
.97619 .3616 L
s
.97157 .4407 m
.97619 .4407 L
s
.97157 .51979 m
.97619 .51979 L
s
.97157 .59888 m
.97619 .59888 L
s
.125 Mabswid
.97342 .1441 m
.97619 .1441 L
s
.97342 .16387 m
.97619 .16387 L
s
.97342 .18364 m
.97619 .18364 L
s
.97342 .22319 m
.97619 .22319 L
s
.97342 .24296 m
.97619 .24296 L
s
.97342 .26274 m
.97619 .26274 L
s
.97342 .30228 m
.97619 .30228 L
s
.97342 .32206 m
.97619 .32206 L
s
.97342 .34183 m
.97619 .34183 L
s
.97342 .38138 m
.97619 .38138 L
s
.97342 .40115 m
.97619 .40115 L
s
.97342 .42092 m
.97619 .42092 L
s
.97342 .46047 m
.97619 .46047 L
s
.97342 .48024 m
.97619 .48024 L
s
.97342 .50002 m
.97619 .50002 L
s
.97342 .53956 m
.97619 .53956 L
s
.97342 .55934 m
.97619 .55934 L
s
.97342 .57911 m
.97619 .57911 L
s
.25 Mabswid
.97619 .11267 m
.97619 .6024 L
s
.5 Mabswid
[ 2.23146 2.23146 ] 0 Mabsdash
.25537 .13124 m
.29051 .44771 L
.32447 .32036 L
.36663 .46844 L
.41088 .41048 L
.45989 .24733 L
.5117 .4797 L
.56707 .43942 L
.62368 .59074 L
.68138 .23492 L
.73975 .46833 L
.79965 .51616 L
.86096 .39353 L
.90979 .39992 L
.95861 .29377 L
s
[ ] 0 Mabsdash
.25537 .13214 m
.29051 .16833 L
.32447 .23476 L
.36663 .23652 L
.41088 .16625 L
.45989 .24367 L
.5117 .24991 L
.56707 .24356 L
.62368 .22336 L
.68138 .13713 L
.73975 .32243 L
.79965 .17756 L
.86096 .27463 L
.90979 .25093 L
.95861 .2575 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[3]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00A`3oool2
000002T0oooo0P0000040?ooo`030000003oool0oooo0080oooo0`00000Z0?ooo`@00000:03oool4
000000<0oooo00<000000?ooo`3oool00P3oool3000002X0oooo1000000=0?ooo`00AP3oool01000
0000oooo0?ooo`00000W0?ooo`040000003oool0oooo000000/0oooo00<000000?ooo`3oool0:@3o
ool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool09`3o
ool00`000000oooo0?ooo`0>0?ooo`00AP3oool010000000oooo0?ooo`00000W0?ooo`040000003o
ool0oooo000000/0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0Y0?ooo`030000
003oool0oooo00X0oooo00<000000?ooo`3oool0:03oool00`000000oooo0?ooo`0=0?ooo`00AP3o
ool010000000oooo0?ooo`00000W0?ooo`040000003oool0oooo000000P0oooo0`00000/0?ooo`03
0000003oool0oooo02T0oooo00<000000?ooo`3oool01`3oool3000002`0oooo00<000000?ooo`3o
ool0303oool004H0oooo00@000000?ooo`3oool000009`3oool010000000oooo0?ooo`0000080?oo
o`030000003oool0oooo02`0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo02d0oooo00<000000?ooo`3oool02`3oool004H0oooo00@000000?ooo`3o
ool000009`3oool010000000oooo0?ooo`0000080?ooo`030000003oool0oooo02X0oooo0`00000Y
0?ooo`<000002@3oool00`000000oooo0?ooo`0Z0?ooo`040000003oool0oooo000000d0oooo0017
0?ooo`800000:@3oool2000000T0oooo1000000[0?ooo`030000003oool0oooo02T0oooo00<00000
0?ooo`3oool01`3oool4000002X0oooo0P00000>0?ooo`00A03ooooD000000P0oooo00140?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool01P3oool004@0oooo00<000000?ooo`3oool0
d03oool00`000000oooo0?ooo`060?ooo`00?@3oool2000000D0oooo0P00003@0?ooo`800000203o
ool003`0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`3@0?ooo`030000003o
ool0oooo00H0oooo000l0?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`3;0?ooo`030000003oool0oooo00H0oooo000l0?ooo`040000003oool0
oooo000000@0oooo00<000000?ooo`3oool00P3oool200000<`0oooo00<000000?ooo`3oool01P3o
ool003`0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`030000003o
ool0000007L0oooo00<000000?ooo`3oool0D@3oool00`000000oooo0?ooo`060?ooo`00?03oool0
10000000oooo0?ooo`0000040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0L`3o
ool2000005<0oooo00<000000?ooo`3oool01P3oool003d0oooo0P0000050?ooo`8000001`3oool0
0`000000oooo0?ooo`1b0?ooo`030000003oool000000580oooo00<000000?ooo`3oool01P3oool0
04@0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo
0700oooo00@000000?ooo`3oool00000DP3oool00`000000oooo0?ooo`060?ooo`00A03oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0K`3oool01@00
0000oooo0?ooo`3oool000000580oooo00<000000?ooo`3oool01P3oool004@0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo06h0oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`1?0?ooo`030000003oool0oooo00H0oooo00140?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool0K03oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo04l0oooo00<000000?ooo`3oool01P3oool004@0oooo0P0000040?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool0JP3oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo04l0oooo00<000000?ooo`3oool01P3oool004@0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0oooo0200oooo00<000000?ooo`3oool0AP3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0oooo04h0oooo00<000000?ooo`3oool01P3o
ool004@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`060?ooo`030000003oool0
oooo01h0oooo00<000000?ooo`000000AP3oool00`000000oooo0?ooo`060?ooo`030000003oool0
oooo04h0oooo00<000000?ooo`3oool01P3oool004@0oooo00<000000?ooo`3oool03@3oool00`00
0000oooo0?ooo`0M0?ooo`030000003oool0000004D0oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`1>0?ooo`030000003oool0oooo00H0oooo00140?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0L0?ooo`050000003oool0oooo0?oo
o`000000A03oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01/0oooo00<000000?oo
o`3oool0;`3oool00`000000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo0400oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0K0?ooo`80
0000<03oool00`000000oooo0?ooo`060?ooo`00A03oool2000000D0oooo00<000000?ooo`3oool0
1`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
?`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`000000
<03oool00`000000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`050?ooo`030000003o
ool0oooo03d0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0J0?ooo`040000003o
ool0oooo000002l0oooo00<000000?ooo`3oool01P3oool004@0oooo00<000000?ooo`3oool03`3o
ool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0?03o
ool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo01P0oooo00D000000?ooo`3oool0oooo
0000000_0?ooo`030000003oool0oooo00H0oooo000F0?ooo`@000000`3oool01@000000oooo0?oo
o`3oool000000080oooo100000030?ooo`8000005`3oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo00T0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`070?ooo`030000
003oool0oooo03/0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0H0?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`060?ooo`005P3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0000000D0oooo00D000000?ooo`3oool0oooo
000000020?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`090?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0203oool00`000000oooo
0?ooo`0i0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0603oool00`000000oooo
0?ooo`030?ooo`030000003oool0oooo02/0oooo00<000000?ooo`3oool01P3oool001L0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
2P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0
>03oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo00H0oooo000H0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`0000001@3oool01@000000oooo0?ooo`3oool000000080oooo
00<000000?ooo`3oool0503oool200000180oooo00<000000?ooo`3oool04`3oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo03L0oooo00<000000?ooo`3oool0403oool00`000000oooo
0?ooo`0F0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0:@3oool2000000P0oooo
000I0?ooo`050000003oool0oooo0?ooo`0000000`3oool00`000000oooo0?ooo`020?ooo`050000
003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00`0
oooo00<000000?ooo`3oool0=@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo01D0
oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo00H0
oooo000F0?ooo`040000003oool0oooo000000X0oooo0`0000030?ooo`040000003oool0oooo0000
01H0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo
0140oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0d0?ooo`030000003oool0oooo
0180oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo
02T0oooo00<000000?ooo`3oool01P3oool001L0oooo0P00000=0?ooo`030000003oool0oooo0080
oooo0P00000G0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool02`3oool00`000000
oooo0?ooo`0@0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0<P3oool00`000000
oooo0?ooo`0C0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0203oool00`000000
oooo0?ooo`0X0?ooo`030000003oool0oooo00H0oooo000g0?ooo`<000000P3oool2000000H0oooo
00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00l0oooo
00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo01@0oooo
00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo02P0oooo
00<000000?ooo`3oool01P3oool003P0oooo00L000000?ooo`3oool0oooo0000003oool0000000D0
oooo0P00000E0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool04P3oool00`000000
oooo0?ooo`0`0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool04P3oool00`000000
oooo0?ooo`0:0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool01P3oool00380oooo
0`0000030?ooo`040000003oool0oooo0?ooo`<000001@3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo00`0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0C0?ooo`03
0000003oool0oooo02d0oooo0P00000H0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00H0oooo000h0?ooo`050000
003oool0oooo0?ooo`0000001`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo02X0
oooo0`00000J0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0303oool00`000000
oooo0?ooo`0V0?ooo`030000003oool0oooo00H0oooo000g0?ooo`8000000`3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool03@3oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool09@3oool4000001h0oooo
00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo02D0oooo
00<000000?ooo`3oool01P3oool003P0oooo00<000000?ooo`3oool00P3oool2000000D0oooo00<0
00000?ooo`3oool05P3oool=000001T0oooo00<000000?ooo`3oool08@3oool3000001T0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00h0oooo00<0
00000?ooo`3oool09@3oool00`000000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo03@0oooo00<000000?ooo`3oool07P3oool3000001`0oooo00<00000
0?ooo`3oool01P3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00l0oooo00<00000
0?ooo`3oool0903oool00`000000oooo0?ooo`060?ooo`00A03oool2000000H0oooo00<000000?oo
o`3oool0=@3oool4000001H0oooo1P00000O0?ooo`030000003oool0oooo00L0oooo00<000000?oo
o`3oool03@3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo02<0oooo00<000000?oo
o`3oool01P3oool004@0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0d0?ooo`04
0000003oool0oooo0?ooo`L000001`3oool8000002D0oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`0=0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool08`3oool00`00
0000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`0m0?ooo`030000003oool0oooo00P0
oooo1`00000/0?ooo`030000003oool0000000P0oooo00<000000?ooo`3oool03@3oool00`000000
oooo0?ooo`0B0?ooo`030000003oool0oooo00h0oooo1@00000?0?ooo`030000003oool0oooo00H0
oooo00140?ooo`030000003oool0oooo03`0oooo00<000000?ooo`3oool0?03oool00`000000oooo
0?ooo`090?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool04`3oool00`000000oooo
0?ooo`0<0?ooo`8000001@3oool7000000P0oooo00<000000?ooo`3oool01P3oool004@0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo0?ooo`0d0?ooo`030000003oool0oooo03/0oooo00@0
00000?ooo`3oool00000203oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo01@0oooo
00<000000?ooo`3oool02@3oool2000000h0oooo100000040?ooo`030000003oool0oooo00H0oooo
00140?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0<P3oool00`000000oooo0000
00100?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?oo
o`0F0?ooo`030000003oool0oooo00H0oooo0P00000H0?ooo`030000003oool0oooo00H0oooo0014
0?ooo`8000001`3oool00`000000oooo0?ooo`0b0?ooo`040000003oool0oooo000003l0oooo00<0
00000?ooo`3oool01`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01H0oooo00<0
00000?ooo`3oool0103oool2000001X0oooo00<000000?ooo`3oool01P3oool004@0oooo00<00000
0?ooo`3oool0>`3oool00`000000oooo0?ooo`0l0?ooo`050000003oool0oooo0?ooo`0000002@3o
ool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01P0oooo00@000000?ooo`3oool0oooo
0P00000L0?ooo`030000003oool0oooo00H0oooo00140?ooo`030000003oool0oooo00H0oooo00<0
00000?ooo`3oool0L@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0603oool010000000oooo00000000000N0?ooo`030000003oool0oooo00H0oooo
000H0?ooo`030000003oool0oooo0080oooo00D000000?ooo`3oool0oooo000000020?ooo`@00000
0`3oool2000001L0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0a0?ooo`050000
003oool0oooo0?ooo`000000>`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo01h0
oooo00<000000?ooo`3oool01P3oool001P0oooo00<000000?ooo`3oool00`3oool00`000000oooo
000000050?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0D0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool0<03oool01@000000oooo0?ooo`3oool00000
0400oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo
03/0oooo00<000000?ooo`3oool01P3oool001D0oooo1@0000050?ooo`030000003oool0oooo00@0
oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo01@0oooo0P00000j0?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool0>03oool00`000000oooo0?ooo`020?oo
o`030000003oool0oooo00L0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0j0?oo
o`800000203oool001D0oooo00@000000?ooo`3oool000001@3oool00`000000oooo000000050?oo
o`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0
oooo07X0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`050?ooo`030000003oool0
oooo03`0oooo00<000000?ooo`3oool01P3oool001H0oooo00<000000?ooo`000000103oool01@00
0000oooo0?ooo`3oool0000000@0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003o
ool0oooo01@0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0e0?ooo`030000003o
ool0oooo03L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`070?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool0?03oool00`000000oooo0?ooo`060?ooo`005`3oool2
000000/0oooo0`0000030?ooo`040000003oool0oooo000001H0oooo00<000000?ooo`3oool01`3o
ool00`000000oooo0?ooo`0_0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0?P3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0?03o
ool00`000000oooo0?ooo`060?ooo`00603oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo0080oooo0P00000G0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0;`3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo03H0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0>03oool0
0`000000oooo0?ooo`020?ooo`030000003oool0oooo00H0oooo000g0?ooo`<000000P3oool20000
00H0oooo00<000000?ooo`3oool0>03oool00`000000oooo0?ooo`0n0?ooo`030000003oool0oooo
00l0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0h0?ooo`030000003oool0oooo
0080oooo00<000000?ooo`3oool01P3oool003P0oooo00L000000?ooo`3oool0oooo0000003oool0
000000D0oooo0P0000080?ooo`030000003oool0oooo03H0oooo00<000000?ooo`3oool0=P3oool0
0`000000oooo0?ooo`050?ooo`030000003oool0oooo00L0oooo00D000000?ooo`3oool0oooo0000
00100?ooo`030000003oool0oooo00H0oooo000b0?ooo`<000000`3oool010000000oooo0?ooo`3o
ool3000000D0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0^0?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool0=P3oool00`000000oooo0?ooo`050?ooo`030000003o
ool0oooo00L0oooo00D000000?ooo`3oool0oooo0000000j0?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool01P3oool003P0oooo00D000000?ooo`3oool0oooo000000070?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`060?ooo`030000
003oool0oooo03d0oooo00<000000?ooo`3oool0203oool010000000oooo0?ooo`00000j0?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool01P3oool003L0oooo0P0000030?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`3oool0=`3oool00`000000oooo0?ooo`0n0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool0203oool00`000000oooo0000000j0?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool01P3oool003P0oooo00<000000?ooo`3oool00P3o
ool2000000D0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`1]0?ooo`030000003o
ool0oooo0140oooo00<000000?ooo`000000@@3oool00`000000oooo0?ooo`060?ooo`00A03oool0
0`000000oooo0?ooo`080?ooo`030000003oool0oooo02/0oooo00<000000?ooo`3oool0203oool0
0`000000oooo0?ooo`0d0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0203oool0
0`000000oooo0?ooo`0h0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01P3oool0
04@0oooo0P0000090?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool07@3oool00`00
0000oooo0?ooo`080?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool0203oool00`00
0000oooo0?ooo`0h0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01P3oool004@0
oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0:0?ooo`8000007`3oool00`000000
oooo0?ooo`080?ooo`030000003oool0oooo03@0oooo00<000000?ooo`3oool01`3oool00`000000
oooo0?ooo`120?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01P3oool004@0oooo
00<000000?ooo`3oool05@3oool200000640oooo00<000000?ooo`3oool0E@3oool00`000000oooo
0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00X0oooo00<0
00000?ooo`3oool07@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0380oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`100?ooo`030000003oool0oooo00L0oooo00<0
00000?ooo`3oool01P3oool004@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0<
0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0n
0?ooo`030000003oool0oooo0400oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`06
0?ooo`00A03oool2000000X0oooo00<000000?ooo`3oool0203oool010000000oooo0?ooo`00000L
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0?P3oool00`000000oooo0?ooo`0o
0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool01P3oool004@0oooo00<000000?oo
o`3oool0503oool01@000000oooo0?ooo`3oool0000005h0oooo00<000000?ooo`3oool0EP3oool0
0`000000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo
00P0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo
00X0oooo00<000000?ooo`3oool0?@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo
00H0oooo00140?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0303oool00`000000
oooo0?ooo`0I0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0<@3oool00`000000
oooo0?ooo`0:0?ooo`030000003oool0oooo03d0oooo00<000000?ooo`3oool02@3oool00`000000
oooo0?ooo`060?ooo`005`3oool2000000@0oooo00D000000?ooo`3oool0oooo000000020?ooo`@0
00000`3oool2000001L0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo01T0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0o0?ooo`03
0000003oool0oooo03/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`060?ooo`00
5P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0000000D0oooo00D000000?ooo`3o
ool0oooo000000020?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`070?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool05`3oool00`00
0000oooo0?ooo`0=0?ooo`030000003oool0oooo0300oooo00<000000?ooo`3oool0303oool00`00
0000oooo0?ooo`180?ooo`030000003oool0oooo00H0oooo000F0?ooo`040000003oool0oooo0000
00D0oooo00<000000?ooo`3oool0103oool01@000000oooo0?ooo`3oool000000080oooo00<00000
0?ooo`3oool0503oool2000001@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1J
0?ooo`030000003oool0oooo04T0oooo00<000000?ooo`3oool02P3oool2000000P0oooo000F0?oo
o`<000001@3oool00`000000oooo000000050?ooo`050000003oool0oooo0?ooo`0000000P3oool0
0`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool01P3oool0
0`000000oooo0?ooo`0M0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0;`3oool0
0`000000oooo0?ooo`0<0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool02`3oool0
0`000000oooo0?ooo`060?ooo`005P3oool00`000000oooo0?ooo`040?ooo`050000003oool0oooo
0?ooo`000000103oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0503o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool05@3o
ool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo03h0oooo00<000000?ooo`3oool0>@3o
ool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00H0oooo000F0?ooo`030000003oool0
oooo00/0oooo0`0000030?ooo`040000003oool0oooo000001H0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool05@3o
ool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo03l0oooo00<000000?ooo`3oool0A`3o
ool00`000000oooo0?ooo`060?ooo`005`3oool3000000`0oooo00<000000?ooo`3oool00P3oool2
000000X0oooo0`0000020?ooo`8000001P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0
oooo00D0oooo00<000000?ooo`3oool0F03oool00`000000oooo0?ooo`180?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool01P3oool003P0oooo00L000000?ooo`3oool0oooo0000003o
ool0000000D0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo01d0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0]0?ooo`030000003o
ool0oooo00h0oooo00<000000?ooo`3oool0=`3oool00`000000oooo0?ooo`0=0?ooo`030000003o
ool0oooo00H0oooo000b0?ooo`<000000`3oool010000000oooo0?ooo`3oool3000000D0oooo0P00
000;0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?oo
o`0A0?ooo`030000003oool0oooo02d0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?oo
o`0g0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool01P3oool003P0oooo00D00000
0?ooo`3oool0oooo000000070?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool04P3o
ool00`000000oooo0?ooo`0o0?ooo`030000003oool0oooo04H0oooo00<000000?ooo`3oool01P3o
ool003L0oooo0P0000030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04@3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo05L0oooo00<000000?ooo`3oool0A`3oool0
0`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00H0oooo000h0?ooo`030000003oool0oooo
0080oooo0P0000050?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool07@3oool00`00
0000oooo0?ooo`0C0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`0W0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool03P3oool00`00
0000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00l0
oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo02/0
oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0U0?ooo`030000003oool0oooo0080
000000<0oooo0000000000000P0000050?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3o
ool01P3oool004@0oooo0P00000;0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0
?`3oool00`000000oooo0?ooo`0T0?ooo`030000003oool0oooo00P0oooo0`00000C0?ooo`030000
003oool0oooo00H0oooo00140?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0503o
ool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool08`3o
ool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo00H0oooo00140?ooo`030000003oool0
oooo00l0oooo00<000000?ooo`3oool0HP3oool00`000000oooo0?ooo`1I0?ooo`030000003oool0
oooo00H0oooo00140?ooo`030000003oool0oooo00/0oooo00D000000?ooo`3oool0oooo0000000N
0?ooo`8000005`3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo0140oooo00<00000
0?ooo`3oool08P3oool00`000000oooo0?ooo`0P0?ooo`030000003oool0oooo00H0oooo00140?oo
o`030000003oool0oooo00/0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0>0?oo
o`030000003oool0oooo01H0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo0280oooo00<000000?ooo`3oool0803oool00`000000oooo0?ooo`060?oo
o`00A03oool00`000000oooo0?ooo`0;0?ooo`040000003oool0oooo000000h0oooo00<000000?oo
o`3oool03@3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo03l0oooo00<000000?oo
o`3oool0803oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo00H0oooo00140?ooo`80
00003`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool0A03oool00`000000oooo0?ooo`1J0?ooo`030000003oool0oooo00H0oooo00140?ooo`030000
003oool0oooo00/0oooo00@000000?ooo`3oool000007@3oool00`000000oooo0?ooo`0I0?ooo`03
0000003oool0oooo02P0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`0O0?ooo`03
0000003oool0oooo0280oooo00<000000?ooo`3oool01P3oool004@0oooo00<000000?ooo`3oool0
2`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0
6P3oool00`000000oooo0?ooo`0X0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool0
7`3oool00`000000oooo0?ooo`0R0?ooo`030000003oool0oooo00H0oooo000G0?ooo`800000103o
ool01@000000oooo0?ooo`3oool000000080oooo100000030?ooo`8000005`3oool00`000000oooo
0?ooo`0<0?ooo`800000403oool00`000000oooo0?ooo`0W0?ooo`030000003oool0oooo03l0oooo
00<000000?ooo`3oool07@3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo00H0oooo
000F0?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`0000001@3oool01@000000oooo
0?ooo`3oool000000080oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`0<0?ooo`80
00004@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo04L0oooo00<000000?ooo`3o
ool0FP3oool00`000000oooo0?ooo`060?ooo`005P3oool010000000oooo0?ooo`0000050?ooo`03
0000003oool0oooo00@0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
01@0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo
01d0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo
01D0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0T0?ooo`030000003oool0oooo
00H0oooo000G0?ooo`8000001@3oool00`000000oooo000000050?ooo`050000003oool0oooo0?oo
o`0000000P3oool00`000000oooo0?ooo`0D0?ooo`8000003@3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0?0?oo
o`030000003oool0oooo01@0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0K0?oo
o`030000003oool0oooo02<0oooo0P0000080?ooo`005P3oool010000000oooo0?ooo`0000030?oo
o`050000003oool0oooo0?ooo`000000103oool01@000000oooo0?ooo`3oool000000080oooo00<0
00000?ooo`3oool0503oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0140oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo00`0oooo0P00
00020?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?oo
o`0U0?ooo`030000003oool0oooo00H0oooo000F0?ooo`040000003oool0oooo000000X0oooo0`00
00030?ooo`040000003oool0oooo000001H0oooo00<000000?ooo`3oool0303oool00`000000oooo
0?ooo`0A0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0803oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo07H0oooo00<000000?ooo`3oool01P3oool001L0oooo0P00
000=0?ooo`030000003oool0oooo0080oooo0P00000G0?ooo`030000003oool0oooo02D0oooo00<0
00000?ooo`3oool0;P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0180oooo00<0
00000?ooo`3oool0<P3oool00`000000oooo0?ooo`0V0?ooo`030000003oool0oooo00H0oooo000g
0?ooo`<000000P3oool2000000H0oooo00<000000?ooo`3oool0BP3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0G
0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool09P3oool00`000000oooo0?ooo`06
0?ooo`00>03oool01`000000oooo0?ooo`3oool000000?ooo`0000001@3oool200000280oooo00@0
00000?ooo`3oool000009@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00H0oooo
00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo01L0oooo
00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`060?ooo`00<P3oool3000000<0oooo00@0
00000?ooo`3oool0oooo0`0000050?ooo`030000003oool0oooo0240oooo00@000000?ooo`3oool0
00009@3oool00`000000oooo0?ooo`040?ooo`8000002P3oool00`000000oooo0?ooo`0/0?ooo`03
0000003oool0oooo0440oooo00<000000?ooo`3oool01P3oool003P0oooo00D000000?ooo`3oool0
oooo000000070?ooo`030000003oool0oooo0280oooo0P00000/0?ooo`030000003oool0oooo01h0
oooo00<000000?ooo`3oool0<@3oool00`000000oooo0?ooo`0X0?ooo`030000003oool0oooo00H0
oooo000g0?ooo`8000000`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo04/0oooo
00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo01T0oooo
00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0X0?ooo`030000003oool0oooo00H0oooo
000h0?ooo`030000003oool0oooo0080oooo0P0000050?ooo`030000003oool0oooo04/0oooo00@0
00000?ooo`3oool000003`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo01X0oooo
0P00000D0?ooo`030000003oool0oooo02T0oooo00<000000?ooo`3oool01P3oool004@0oooo00<0
00000?ooo`3oool0B`3oool00`000000oooo0000000A0?ooo`030000003oool0oooo06h0oooo00<0
00000?ooo`3oool01P3oool004@0oooo0P00001a0?ooo`030000003oool0oooo01d0oooo00<00000
0?ooo`3oool0403oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo00H0oooo00140?oo
o`030000003oool0oooo0600oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0N0?oo
o`800000403oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo00H0oooo00140?ooo`03
0000003oool0oooo0600oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0Q0?ooo`03
0000003oool0oooo00/0oooo00<000000?ooo`3oool0:`3oool00`000000oooo0?ooo`060?ooo`00
A03oool00`000000oooo0?ooo`1Q0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0
8P3oool00`000000oooo0?ooo`0h0?ooo`030000003oool0oooo00H0oooo00140?ooo`030000003o
ool0oooo09H0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0/0?ooo`030000003o
ool0oooo00H0oooo00140?ooo`030000003oool0oooo0640oooo00<000000?ooo`3oool0303oool0
0`000000oooo0?ooo`0T0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0;03oool0
0`000000oooo0?ooo`060?ooo`00A03oool200000680oooo00<000000?ooo`3oool0303oool00`00
0000oooo0?ooo`0U0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0;@3oool00`00
0000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`1R0?ooo`030000003oool0oooo00X0
oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`0d0?ooo`030000003oool0oooo00H0
oooo00140?ooo`030000003oool0oooo09/0oooo00D000000?ooo`3oool0oooo0000000`0?ooo`03
0000003oool0oooo00H0oooo000F0?ooo`@000000`3oool01@000000oooo0?ooo`3oool000000080
oooo100000030?ooo`8000005`3oool00`000000oooo0?ooo`1S0?ooo`030000003oool0oooo00T0
oooo00<000000?ooo`3oool0:P3oool2000000030?ooo`000000oooo02l0oooo00<000000?ooo`3o
ool01P3oool001P0oooo00<000000?ooo`3oool00`3oool00`000000oooo000000050?ooo`050000
003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo06<0
oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0/0?ooo`030000003oool0oooo02l0
oooo00<000000?ooo`3oool01P3oool001P0oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`040?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0D0?ooo`80
0000I@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo05d0oooo0P0000080?ooo`00
603oool00`000000oooo0?ooo`030?ooo`030000003oool0000000D0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0d03oool00`000000
oooo0?ooo`060?ooo`00603oool00`000000oooo0?ooo`020?ooo`050000003oool0oooo0?ooo`00
0000103oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0503oool00`00
0000oooo0?ooo`1U0?ooo`030000003oool0oooo06P0oooo00<000000?ooo`3oool01P3oool001H0
oooo0`00000;0?ooo`<000000`3oool010000000oooo0?ooo`00000F0?ooo`030000003oool0oooo
06D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`1O0?ooo`030000003oool0oooo
00H0oooo000H0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool00P3oool2000001L0
oooo00<000000?ooo`3oool0I@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo05l0
oooo00<000000?ooo`3oool01P3oool003L0oooo0`0000020?ooo`8000001P3oool00`000000oooo
0?ooo`1^0?ooo`030000003oool0oooo05l0oooo00<000000?ooo`3oool01P3oool003P0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`800000d@3oool00`000000oooo0?oo
o`060?ooo`00<P3oool3000000<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`03
0?ooo`030000003oool0oooo06H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`1O
0?ooo`030000003oool0oooo00H0oooo000h0?ooo`040000003oool0oooo0?ooo`8000001P3oool0
0`000000oooo0?ooo`1V0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0G`3oool0
0`000000oooo0?ooo`060?ooo`00=`3oool2000000<0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`1W0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0H03oool00`00
0000oooo0?ooo`060?ooo`00>03oool010000000oooo0?ooo`3oool3000000D0oooo00<000000?oo
o`3oool0d03oool00`000000oooo0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`1X0?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool0H03oool00`000000oooo0?ooo`060?ooo`00
A03oool2000006T0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`1P0?ooo`030000
003oool0oooo00H0oooo00140?ooo`030000003oool0oooo06P0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`1P0?ooo`030000003oool0oooo00H0oooo00140?ooo`030000003oool0
oooo06d0oooo00<000000?ooo`3oool0H03oool00`000000oooo0?ooo`060?ooo`00A03oool00`00
0000oooo0?ooo`1Y0?ooo`030000003oool0oooo06@0oooo00<000000?ooo`3oool01P3oool004@0
oooo00<000000?ooo`3oool0J@3oool010000000oooo0?ooo`00001S0?ooo`030000003oool0oooo
00H0oooo00140?ooo`800000J`3oool00`000000oooo0000001S0?ooo`030000003oool0oooo00H0
oooo00140?ooo`030000003oool0oooo06`0oooo00<000000?ooo`3oool0H@3oool00`000000oooo
0?ooo`060?ooo`00A03oool00`000000oooo0?ooo`1[0?ooo`030000003oool0oooo0680oooo00<0
00000?ooo`3oool01P3oool004@0oooo00<000000?ooo`3oool0J`3oool00`000000oooo0?ooo`1R
0?ooo`030000003oool0oooo00H0oooo00080?ooo`@000000`3oool00`000000oooo0?ooo`020?oo
o`@000001@3oool01@000000oooo0?ooo`3oool000000080oooo100000030?ooo`8000005`3oool0
0`000000oooo0?ooo`1[0?ooo`030000003oool0oooo0680oooo00<000000?ooo`3oool01P3oool0
00X0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`070?ooo`030000003oool00000
00D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo01@0oooo00<00000
0?ooo`3oool0d03oool00`000000oooo0?ooo`060?ooo`002P3oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo00L0oooo00<000000?ooo`3oool0103oool01@000000oooo0?ooo`3oool0
00000080oooo00<000000?ooo`3oool0503oool2000000<0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`070?ooo`040000003oool0oooo0?ooo`800000203oool000X0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`050?ooo`030000003oool0000000D0oooo00D00000
0?ooo`3oool0oooo000000020?ooo`030000003oool0oooo01@0ooooe00000080?ooo`002P3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00<0oooo00D000000?ooo`3oool0oooo0000
00040?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`3`0?ooo`00203o
ool3000000T0oooo00@000000?ooo`3oool00000303oool3000000<0oooo00@000000?ooo`3oool0
0000lP3oool000X0oooo00<000000?ooo`3oool0203oool2000000l0oooo00<000000?ooo`3oool0
0P3oool2000000X0oooo0`0000020?ooo`800000hP3oool003P0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`3O0?ooo`00\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {494.25, 317.312}} -> {22.2783, \
246.38, 0.844462, 0.786909}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Rectangular systems", "Subsection"],

Cell[TextData[{
  "In the following example it is shown how the implementation of the \
orthogonal projection method also works for rectangular matrix differential \
systems. Formally stated, we are interested in solving ordinary differential \
equations on the ",
  StyleBox["Stiefel manifold",
    FontSlant->"Italic"],
  ", the set of n\[Times]p orthogonal matrices with p < n."
}], "Text"],

Cell[TextData[{
  StyleBox["Definition",
    FontWeight->"Bold"],
  " The ",
  StyleBox["Stiefel manifold",
    FontSlant->"Italic"],
  " of n\[Times]p orthogonal matrices is the set ",
  Cell[BoxData[
      \(TraditionalForm\`\(V\_\(n, p\)\)(\[DoubleStruckCapitalR])\)]],
  " = {Y\[Element]",
  Cell[BoxData[
      \(TraditionalForm\`\[DoubleStruckCapitalR]\^\(n\[Times]p\)\)]],
  " | ",
  Cell[BoxData[
      \(TraditionalForm\`Y\^T\ Y\  = \ I\_p\)]],
  "}, 1\[LessEqual]p<n, where ",
  Cell[BoxData[
      \(TraditionalForm\`I\_p\)]],
  "is the p\[Times]p identity matrix."
}], "Text"],

Cell["\<\
Solutions that evolve on the Stiefel manifold find numerous \
applications such as eigenvalue problems in numerical linear algebra, \
computation of Lyapunov exponents for dynamical systems and signal \
processing.\
\>", "Text"],

Cell[TextData[{
  "Consider an example adapted from [",
  ButtonBox["DL01",
    ButtonData:>{"References.nb", "DL01"},
    ButtonStyle->"Hyperlink"],
  "]:"
}], "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{\(\(q\^\[Prime]\)(t) = A\ \(q(t)\)\), ",", \(t > 0\), ",", 
        RowBox[{\(q(0)\), "=", 
          FormBox[\(q\_0\),
            "TraditionalForm"]}]}], TraditionalForm]], "DisplayFormula"],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`q\_0\  = \ 1/\(\@n\)\ [1, \[Ellipsis], 1]\^T\)]],
  " , A = ",
  Cell[BoxData[
      \(TraditionalForm\`diag[a\_1, \[Ellipsis], 
          a\_n] \[Element] \[DoubleStruckCapitalR]\^\(n\[Times]n\)\)]],
  ", with ",
  Cell[BoxData[
      \(TraditionalForm\`a\_i = \ \(\((\(-1\))\)\^i\) \[Alpha]\ , \ 
      i = 1, \[Ellipsis], n\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Alpha] > 0\)]],
  "."
}], "Text"],

Cell["The exact solution is given by:", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{\(q(t)\), " ", "=", " ", 
        RowBox[{\(1\/\@n\), 
          RowBox[{
            TagBox[
              RowBox[{"(", "\[NoBreak]", GridBox[{
                    {\(exp(\(a\_1\) t)\)},
                    {"\[VerticalEllipsis]"},
                    {\(exp(\(a\_n\) t)\)}
                    }], "\[NoBreak]", ")"}],
              Function[ BoxForm`e$, 
                MatrixForm[ BoxForm`e$]]], "."}]}]}], 
      TraditionalForm]], "DisplayFormula"],

Cell["Normalizing q(t) as:", "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{\(Y(t)\), " ", "=", " ", 
        RowBox[{\(\(q(t)\)\/\(\(||\)\(q(t)\)\(||\)\)\), 
          
          TagBox[\(\(\[Element]\)\(\[DoubleStruckCapitalR]\^\(n\[Times]1\)\)\)\
,
            Function[ BoxForm`e$, 
              MatrixForm[ BoxForm`e$]]]}]}], 
      TraditionalForm]], "DisplayFormula"],

Cell[TextData[{
  "it follows that Y(t) satisfies the following weak skew-symmetric system on \
",
  Cell[BoxData[
      \(TraditionalForm\`\(V\_\(n, \ 1\)\)(\[DoubleStruckCapitalR])\)]],
  ":"
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {\(Y\^\[Prime]\), "=", \(\(F(Y)\)\ Y\)},
          {" ", "=", \(\((I\_n - \ Y\ Y\^T)\)\ A\ Y\)}
          }], TraditionalForm]], "DisplayFormula"],

Cell["\<\
In the following example, the system is solved on the interval [0, \
5] with \[Alpha] = 9/10 and dimension n = 2.\
\>", "Text"],

Cell[BoxData[{
    \(p\  = \ 1; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(n\  = \ 2; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(\[Alpha]\  = \ 9\/10; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(ics\  = \ \(1\/\@n\) 
        Table[1, {n}]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(avec\  = \ 
      Table[\((\(-1\))\)\^i\ \[Alpha], {i, \ 
          n}]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(A\  = \ 
      DiagonalMatrix[avec]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(Y = 
      Table[\(y[i, 1]\)[t], {i, \ n}, {j, 
          p}]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(F\  = \ \((IdentityMatrix[Length[Y]]\  - Y . Transpose[Y])\) . 
        A; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(system = 
      Thread[Flatten[D[Y, t]] \[Equal] 
          Flatten[F . Y]]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(ics = 
      Thread[Flatten[\((Y /. t \[Rule] 0)\)] \[Equal] 
          ics]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(eqs = {system, ics}; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(vars\  = \ Flatten[Y]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(tfinal\  = \ 5. ; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(time = {t, 0, tfinal}; \)}], "Input",
  CellLabel->"In[21]:="],

Cell["\<\
This computes the exact solution which can be evaluated throughout \
the integration interval.\
\>", "Text"],

Cell[BoxData[
    \(solexact\  = \ \(Transpose[{\((#\/Norm[#, \ 2])\)}] &\)\ @\ \((Exp[avec\
\ t]\/\@n\ )\); \)], "Input",
  CellLabel->"In[35]:="],

Cell["\<\
This computes the solution using an explicit Runge-Kutta \
method.\
\>", "Text"],

Cell[BoxData[{
    \(solerk = 
      NDSolve[eqs, vars, time, 
        Method \[Rule] "\<ExplicitRungeKutta\>"]; \[IndentingNewLine]\), "\
\[IndentingNewLine]", 
    \(solerk\  = \ Y\  /. \ First[solerk]; \)}], "Input",
  CellLabel->"In[36]:="],

Cell["\<\
This computes the componentwise absolute global error at the end of \
the integration interval.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((solexact\  - solerk)\)\  /. \ t \[Rule] tfinal\)], "Input",
  CellLabel->"In[38]:="],

Cell[BoxData[
    \({{\(-2.0340662016055458`*^-11\)}, {2.963185252724543`*^-13}}\)], \
"Output",
  CellLabel->"Out[38]="]
}, Open  ]],

Cell["\<\
This computes the orthogonal error - a measure of the deviation \
from the Stiefel manifold.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(OrthogonalErrorPlot[solerk]; \)], "Input",
  CellLabel->"In[39]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.190476 0.0147151 6.94846e+09 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.21429 -0.0125 -3 -9 ]
[.21429 -0.0125 3 0 ]
[.40476 -0.0125 -3 -9 ]
[.40476 -0.0125 3 0 ]
[.59524 -0.0125 -3 -9 ]
[.59524 -0.0125 3 0 ]
[.78571 -0.0125 -3 -9 ]
[.78571 -0.0125 3 0 ]
[.97619 -0.0125 -3 -9 ]
[.97619 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01472 -6 -4.5 ]
[-0.0125 .01472 0 4.5 ]
[-0.0125 .15368 -37.5 -6.25 ]
[-0.0125 .15368 0 6.25 ]
[-0.0125 .29265 -37.5 -6.25 ]
[-0.0125 .29265 0 6.25 ]
[-0.0125 .43162 -37.5 -6.25 ]
[-0.0125 .43162 0 6.25 ]
[-0.0125 .57059 -37.5 -6.25 ]
[-0.0125 .57059 0 6.25 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[.5 .63053 -113.125 0 ]
[.5 .63053 113.125 13.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(1)] .21429 -0.0125 0 1 Mshowa
.40476 0 m
.40476 .00625 L
s
[(2)] .40476 -0.0125 0 1 Mshowa
.59524 0 m
.59524 .00625 L
s
[(3)] .59524 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(4)] .78571 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(5)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.0619 0 m
.0619 .00375 L
s
.1 0 m
.1 .00375 L
s
.1381 0 m
.1381 .00375 L
s
.17619 0 m
.17619 .00375 L
s
.25238 0 m
.25238 .00375 L
s
.29048 0 m
.29048 .00375 L
s
.32857 0 m
.32857 .00375 L
s
.36667 0 m
.36667 .00375 L
s
.44286 0 m
.44286 .00375 L
s
.48095 0 m
.48095 .00375 L
s
.51905 0 m
.51905 .00375 L
s
.55714 0 m
.55714 .00375 L
s
.63333 0 m
.63333 .00375 L
s
.67143 0 m
.67143 .00375 L
s
.70952 0 m
.70952 .00375 L
s
.74762 0 m
.74762 .00375 L
s
.82381 0 m
.82381 .00375 L
s
.8619 0 m
.8619 .00375 L
s
.9 0 m
.9 .00375 L
s
.9381 0 m
.9381 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01472 m
.00625 .01472 L
s
[(0)] -0.0125 .01472 1 0 Mshowa
0 .15368 m
.00625 .15368 L
s
gsave
-0.0125 .15368 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .29265 m
.00625 .29265 L
s
gsave
-0.0125 .29265 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .43162 m
.00625 .43162 L
s
gsave
-0.0125 .43162 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(6) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
0 .57059 m
.00625 .57059 L
s
gsave
-0.0125 .57059 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(8) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-11) show
grestore
.125 Mabswid
0 .04946 m
.00375 .04946 L
s
0 .0842 m
.00375 .0842 L
s
0 .11894 m
.00375 .11894 L
s
0 .18843 m
.00375 .18843 L
s
0 .22317 m
.00375 .22317 L
s
0 .25791 m
.00375 .25791 L
s
0 .3274 m
.00375 .3274 L
s
0 .36214 m
.00375 .36214 L
s
0 .39688 m
.00375 .39688 L
s
0 .46636 m
.00375 .46636 L
s
0 .50111 m
.00375 .50111 L
s
0 .53585 m
.00375 .53585 L
s
0 .60533 m
.00375 .60533 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.02381 .61178 m
.02381 .61803 L
s
.21429 .61178 m
.21429 .61803 L
s
.40476 .61178 m
.40476 .61803 L
s
.59524 .61178 m
.59524 .61803 L
s
.78571 .61178 m
.78571 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.0619 .61428 m
.0619 .61803 L
s
.1 .61428 m
.1 .61803 L
s
.1381 .61428 m
.1381 .61803 L
s
.17619 .61428 m
.17619 .61803 L
s
.25238 .61428 m
.25238 .61803 L
s
.29048 .61428 m
.29048 .61803 L
s
.32857 .61428 m
.32857 .61803 L
s
.36667 .61428 m
.36667 .61803 L
s
.44286 .61428 m
.44286 .61803 L
s
.48095 .61428 m
.48095 .61803 L
s
.51905 .61428 m
.51905 .61803 L
s
.55714 .61428 m
.55714 .61803 L
s
.63333 .61428 m
.63333 .61803 L
s
.67143 .61428 m
.67143 .61803 L
s
.70952 .61428 m
.70952 .61803 L
s
.74762 .61428 m
.74762 .61803 L
s
.82381 .61428 m
.82381 .61803 L
s
.8619 .61428 m
.8619 .61803 L
s
.9 .61428 m
.9 .61803 L
s
.9381 .61428 m
.9381 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
.99375 .15368 m
1 .15368 L
s
.99375 .29265 m
1 .29265 L
s
.99375 .43162 m
1 .43162 L
s
.99375 .57059 m
1 .57059 L
s
.125 Mabswid
.99625 .04946 m
1 .04946 L
s
.99625 .0842 m
1 .0842 L
s
.99625 .11894 m
1 .11894 L
s
.99625 .18843 m
1 .18843 L
s
.99625 .22317 m
1 .22317 L
s
.99625 .25791 m
1 .25791 L
s
.99625 .3274 m
1 .3274 L
s
.99625 .36214 m
1 .36214 L
s
.99625 .39688 m
1 .39688 L
s
.99625 .46636 m
1 .46636 L
s
.99625 .50111 m
1 .50111 L
s
.99625 .53585 m
1 .53585 L
s
.99625 .60533 m
1 .60533 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
gsave
.5 .63053 -174.125 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.5 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 12.812 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Orthogonal) show
129.000 12.812 moveto
(error) show
165.000 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
177.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
183.000 9.000 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(T) show
189.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
201.125 12.812 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
213.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(I) show
219.125 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
231.125 16.500 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(F) show
243.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(vs) show
261.250 12.812 moveto
(time) show
285.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .01472 m
.04624 .521 L
.06032 .5161 L
.07911 .60332 L
.0965 .5957 L
.11523 .55727 L
.13448 .46273 L
.15391 .33804 L
.17399 .19512 L
.19496 .04332 L
.21736 .15926 L
.24074 .30627 L
.26499 .41043 L
.29042 .48448 L
.31618 .49482 L
.34252 .47264 L
.36945 .43106 L
.3971 .38096 L
.42553 .32847 L
.45482 .27769 L
.48505 .23089 L
.51632 .18928 L
.54871 .15327 L
.58233 .12286 L
.61728 .0977 L
.65369 .07729 L
.69169 .06106 L
.73141 .04838 L
.77304 .03868 L
.81674 .03141 L
.86274 .02607 L
.91127 .02224 L
.94373 .02026 L
.97619 .0188 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[39]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00<03oool2000002/0oooo1000000Y0?ooo`@00000:`3oool2000002d0
oooo00<000000?ooo`3oool0:@3oool3000000T0oooo000_0?ooo`040000003oool0oooo000002`0
oooo00<000000?ooo`3oool0:03oool00`000000oooo0?ooo`0[0?ooo`040000003oool0oooo0000
02`0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`060?ooo`00;`3oool010000000
oooo0?ooo`00000/0?ooo`030000003oool0oooo02T0oooo00<000000?ooo`3oool0;@3oool00`00
0000oooo0?ooo`0W0?ooo`D00000;@3oool00`000000oooo0?ooo`060?ooo`00;`3oool010000000
oooo0?ooo`00000/0?ooo`030000003oool0oooo02X0oooo00<000000?ooo`3oool0:P3oool20000
02X0oooo00@000000?ooo`3oool00000:`3oool3000000T0oooo000_0?ooo`040000003oool0oooo
000002`0oooo00<000000?ooo`3oool0:`3oool00`000000oooo0?ooo`0[0?ooo`030000003oool0
oooo02P0oooo00<000000?ooo`000000:`3oool00`000000oooo0?ooo`090?ooo`00;`3oool01000
0000oooo0?ooo`00000Z0?ooo`<00000:P3oool010000000oooo0?ooo`00000Z0?ooo`040000003o
ool0oooo000002/0oooo0P00000[0?ooo`030000003oool0oooo00T0oooo000`0?ooo`800000;@3o
ool00`000000oooo0?ooo`0Y0?ooo`800000;03oool2000002d0oooo00<000000?ooo`3oool0:@3o
ool4000000P0oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo000[0?oooo400000
103oool002@0oooo0P0000050?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0:`3o
ool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo02/0oooo00<000000?ooo`3oool0:`3o
ool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3o
ool002<0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`3]0?ooo`030000003o
ool0oooo0080oooo000S0?ooo`040000003oool0oooo000000@0oooo0`0000030?ooo`030000003o
ool0oooo0>H0oooo0P0000040?ooo`008`3oool010000000oooo0?ooo`0000040?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool0eP3oool=000000@0oooo00<000000?ooo`3oool00P3o
ool002<0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo0<`0oooo2P00000A0?ooo`030000003oool0oooo0080oooo000S0?ooo`040000003oool0
oooo000000@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`310?ooo`/000006`3o
ool00`000000oooo0?ooo`020?ooo`00903oool2000000D0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`2i0?ooo`P000009P3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo0;@0oooo1@00000^0?ooo`030000003oool0oooo
0080oooo000[0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0[`3oool5000003<0
oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`0V0?ooo`030000003oool0oooo0840oooo1@00000h0?ooo`030000003oool0oooo0080oooo
000[0?ooo`800000103oool00`000000oooo0?ooo`0V0?ooo`030000003oool0oooo07d0oooo1000
000l0?ooo`800000103oool002/0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0V
0?ooo`030000003oool0oooo07T0oooo100000110?ooo`030000003oool0oooo0080oooo000[0?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool09P3oool2000007L0oooo0`0000150?oo
o`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool09@3oool00`000000oooo0000001d0?ooo`<00000B03oool00`000000oooo0?ooo`020?ooo`00
:`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo02D0oooo00<000000?ooo`000000
LP3oool2000004/0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`0T0?ooo`030000003oool000000700oooo0P00001=0?ooo`030000003o
ool0oooo0080oooo000[0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0903oool0
0`000000oooo0000001^0?ooo`800000C`3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo02@0oooo00<000000?ooo`000000K03oool20000
0540oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`0T0?ooo`040000003oool0oooo000006T0oooo0P00001C0?ooo`030000003oool0oooo
0080oooo000[0?ooo`8000001@3oool00`000000oooo0?ooo`0T0?ooo`040000003oool0oooo0000
06L0oooo0P00001D0?ooo`800000103oool002/0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`0S0?ooo`050000003oool0oooo0?ooo`000000I@3oool2000005L0oooo00<000000?oo
o`3oool00P3oool002/0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0S0?ooo`05
0000003oool0oooo0?ooo`000000H`3oool2000005T0oooo00<000000?ooo`3oool00P3oool002/0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0S0?ooo`050000003oool0oooo0?oo
o`000000HP3oool00`000000oooo0?ooo`1I0?ooo`030000003oool0oooo0080oooo000[0?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool08`3oool01@000000oooo0?ooo`3oool00000
0600oooo0P00001L0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool08`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo05`0
oooo00<000000?ooo`3oool0G03oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo
0?ooo`040?ooo`030000003oool0oooo02<0oooo00<000000?ooo`3oool00P3oool00`000000oooo
0?ooo`1K0?ooo`030000003oool0oooo05d0oooo00<000000?ooo`3oool00P3oool002/0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool0F@3oool200000600oooo00<000000?ooo`3oool00P3oool002/0oooo0P000005
0?ooo`030000003oool0oooo02<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`1H
0?ooo`030000003oool0oooo05l0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo0280oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1G0?oo
o`030000003oool0oooo0640oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`0R0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3o
ool0E@3oool00`000000oooo0?ooo`1R0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool08P3oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo05@0oooo00<000000?ooo`3oool0H`3oool00`000000oooo0?ooo`020?ooo`00:`3o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0280oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`1B0?ooo`800000IP3oool00`000000oooo0?ooo`020?ooo`00103oool4
00000080oooo00D000000?ooo`3oool0oooo000000060?ooo`@000000`3oool2000000d0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`0R0?ooo`030000003oool0oooo00@0oooo00<0
00000?ooo`3oool0D@3oool00`000000oooo0?ooo`1V0?ooo`030000003oool0oooo0080oooo0004
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3o
ool000000080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo0280oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`1@0?ooo`030000003o
ool0oooo06L0oooo00<000000?ooo`3oool00P3oool000D0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool08P3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo04l0oooo00<000000?ooo`3oool0J03oool00`000000oooo0?oo
o`020?ooo`001P3oool00`000000oooo0?ooo`020?ooo`030000003oool0000000T0oooo00D00000
0?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo0`0000040?ooo`030000003o
ool0oooo0240oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`1=0?ooo`030000003o
ool0oooo06P0oooo0P0000040?ooo`001`3oool010000000oooo0?ooo`0000030?ooo`030000003o
ool0oooo00H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo00H0oooo
00<000000?ooo`3oool0C03oool00`000000oooo0?ooo`1Z0?ooo`030000003oool0oooo0080oooo
00040?ooo`040000003oool0oooo000000d0oooo0`0000030?ooo`040000003oool0oooo000000L0
oooo0`0000020?ooo`<00000103oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo00H0
oooo00<000000?ooo`3oool0B`3oool00`000000oooo0?ooo`1[0?ooo`030000003oool0oooo0080
oooo00050?ooo`800000403oool00`000000oooo0?ooo`020?ooo`8000002@3oool010000000oooo
0?ooo`3oool2000000D0oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo04X0oooo00<000000?ooo`3oool0K03oool00`000000oooo0?ooo`020?ooo`00
8@3oool3000000<0oooo00@000000?ooo`3oool0oooo0P0000060?ooo`030000003oool0oooo0200
oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`190?ooo`030000003oool0oooo06d0
oooo00<000000?ooo`3oool00P3oool002L0oooo00@000000?ooo`3oool0oooo0P0000060?ooo`03
0000003oool0oooo0200oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`180?ooo`03
0000003oool0oooo06d0oooo00<000000?ooo`3oool00P3oool002H0oooo0P0000030?ooo`800000
1P3oool00`000000oooo0?ooo`0P0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0
A`3oool00`000000oooo0?ooo`1^0?ooo`030000003oool0oooo0080oooo000W0?ooo`040000003o
ool0oooo0?ooo`8000001P3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo00P0oooo
00<000000?ooo`3oool0AP3oool00`000000oooo0?ooo`1_0?ooo`030000003oool0oooo0080oooo
000[0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool07`3oool00`000000oooo0?oo
o`080?ooo`030000003oool0oooo04D0oooo00<000000?ooo`3oool0L03oool00`000000oooo0?oo
o`020?ooo`00:`3oool2000000H0oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo04@0oooo00<000000?ooo`3oool0L03oool2000000@0oooo000[0?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo04<0oooo00<000000?ooo`3oool0LP3oool00`000000oooo0?ooo`020?oo
o`00:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01l0oooo00<000000?ooo`3o
ool02@3oool00`000000oooo0?ooo`120?ooo`030000003oool0oooo0780oooo00<000000?ooo`3o
ool00P3oool002/0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0O0?ooo`030000
003oool0oooo00T0oooo00<000000?ooo`3oool0@@3oool00`000000oooo0?ooo`1c0?ooo`030000
003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool07`3o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0400oooo00<000000?ooo`3oool0M03o
ool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0
oooo01h0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`100?ooo`030000003oool0
oooo07@0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`0N0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0?`3oool00`00
0000oooo0?ooo`1e0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo03h0
oooo00<000000?ooo`3oool0MP3oool00`000000oooo0?ooo`020?ooo`00:`3oool2000000H0oooo
00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo03`0oooo
00<000000?ooo`3oool0MP3oool2000000@0oooo000[0?ooo`030000003oool0oooo00D0oooo00<0
00000?ooo`3oool07P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo03`0oooo00<0
00000?ooo`3oool0M`3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0k
0?ooo`030000003oool0oooo07P0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?oo
o`3oool01@3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo00/0oooo00<000000?oo
o`3oool0>P3oool00`000000oooo0?ooo`1i0?ooo`030000003oool0oooo0080oooo000[0?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool07@3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo03X0oooo00<000000?ooo`3oool0N@3oool00`000000oooo0?ooo`020?ooo`00
:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3oool0
303oool00`000000oooo0?ooo`0i0?ooo`030000003oool0oooo07X0oooo00<000000?ooo`3oool0
0P3oool002/0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0M0?ooo`030000003o
ool0oooo00d0oooo00<000000?ooo`3oool0=`3oool00`000000oooo0?ooo`1k0?ooo`030000003o
ool0oooo0080oooo000[0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool07@3oool0
0`000000oooo0?ooo`0=0?ooo`030000003oool0oooo03H0oooo00<000000?ooo`3oool0O03oool0
0`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
01d0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0f0?ooo`030000003oool0oooo
07`0oooo00<000000?ooo`3oool00P3oool002/0oooo0P0000060?ooo`030000003oool0oooo01d0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0e0?ooo`030000003oool0oooo07`0
oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01d0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0d0?ooo`030000003oool0oooo07h0oooo
00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?oo
o`0L0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0<`3oool00`000000oooo0?oo
o`1o0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00H0oooo00<00000
0?ooo`3oool06`3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo0380oooo00<00000
0?ooo`3oool0O`3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`060?oo
o`030000003oool0oooo01/0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0a0?oo
o`030000003oool0oooo0800oooo00<000000?ooo`3oool00P3oool000H0oooo00D000000?ooo`3o
ool0oooo000000030?ooo`030000003oool0oooo00@0oooo100000030?ooo`8000003@3oool00`00
0000oooo0?ooo`060?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`0`0?ooo`030000003oool0oooo0840oooo00<000000?ooo`3oool00P3oool000H0
oooo00<000000?ooo`3oool00P3oool00`000000oooo000000090?ooo`050000003oool0oooo0?oo
o`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool06`3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo0300oooo00<000000?oo
o`3oool0P@3oool00`000000oooo0?ooo`020?ooo`000`3oool5000000@0oooo00<000000?ooo`3o
ool0203oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`060?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`0_0?ooo`030000003oool0oooo0880oooo00<000000?ooo`3oool00P3oool000<0
oooo00@000000?ooo`3oool00000103oool00`000000oooo000000090?ooo`050000003oool0oooo
0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`<000001P3oool00`000000oooo0?ooo`0K
0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0;`3oool00`000000oooo0?ooo`21
0?ooo`800000103oool000@0oooo00<000000?ooo`0000000`3oool01@000000oooo0?ooo`3oool0
000000P0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo0140oooo00<0
00000?ooo`3oool0;@3oool00`000000oooo0?ooo`230?ooo`030000003oool0oooo0080oooo0005
0?ooo`8000003P3oool3000000<0oooo00@000000?ooo`3oool000001`3oool300000080oooo0`00
00060?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?oo
o`0]0?ooo`030000003oool0oooo08<0oooo00<000000?ooo`3oool00P3oool000H0oooo00<00000
0?ooo`3oool03P3oool00`000000oooo0?ooo`020?ooo`8000002@3oool010000000oooo0?ooo`3o
ool2000000L0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`0A0?ooo`030000003o
ool0oooo02`0oooo00<000000?ooo`3oool0Q03oool00`000000oooo0?ooo`020?ooo`008@3oool3
000000<0oooo00@000000?ooo`3oool0oooo0P0000070?ooo`030000003oool0oooo01X0oooo00<0
00000?ooo`3oool04@3oool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo08D0oooo00<0
00000?ooo`3oool00P3oool002L0oooo00@000000?ooo`3oool0oooo0P0000070?ooo`030000003o
ool0oooo01X0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0[0?ooo`030000003o
ool0oooo08D0oooo00<000000?ooo`3oool00P3oool002H0oooo0P0000030?ooo`8000001`3oool0
0`000000oooo0?ooo`0J0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0:@3oool0
0`000000oooo0?ooo`260?ooo`030000003oool0oooo0080oooo000W0?ooo`040000003oool0oooo
0?ooo`8000001`3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo0180oooo00<00000
0?ooo`3oool0:@3oool00`000000oooo0?ooo`260?ooo`030000003oool0oooo0080oooo000[0?oo
o`8000001`3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo01<0oooo00<000000?oo
o`3oool0:03oool00`000000oooo0?ooo`260?ooo`800000103oool002/0oooo00<000000?ooo`3o
ool01P3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3o
ool09`3oool00`000000oooo0?ooo`280?ooo`030000003oool0oooo0080oooo000[0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`0D0?ooo`030000
003oool0oooo02H0oooo00<000000?ooo`3oool0R03oool00`000000oooo0?ooo`020?ooo`00:`3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0503o
ool00`000000oooo0?ooo`0U0?ooo`030000003oool0oooo08T0oooo00<000000?ooo`3oool00P3o
ool002/0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0
oooo01@0oooo00<000000?ooo`3oool09@3oool00`000000oooo0?ooo`290?ooo`030000003oool0
oooo0080oooo000[0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool06@3oool00`00
0000oooo0?ooo`0D0?ooo`030000003oool0oooo02@0oooo00<000000?ooo`3oool0RP3oool00`00
0000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01T0
oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo08X0
oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`0H0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool08P3oool00`000000oooo
0?ooo`2;0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00H0oooo00<0
00000?ooo`3oool0603oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo0280oooo00<0
00000?ooo`3oool0R`3oool00`000000oooo0?ooo`020?ooo`00:`3oool2000000L0oooo00<00000
0?ooo`3oool0603oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo0240oooo00<00000
0?ooo`3oool0R`3oool2000000@0oooo000[0?ooo`030000003oool0oooo00H0oooo00<000000?oo
o`3oool0603oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo0200oooo00<000000?oo
o`3oool0S03oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo01L0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`0O0?ooo`03
0000003oool0oooo08d0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0
1`3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0
7`3oool00`000000oooo0?ooo`2=0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003o
ool0oooo00L0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0H0?ooo`030000003o
ool0oooo01h0oooo00<000000?ooo`3oool0SP3oool00`000000oooo0?ooo`020?ooo`00:`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool06@3oool0
0`000000oooo0?ooo`0M0?ooo`030000003oool0oooo08h0oooo00<000000?ooo`3oool00P3oool0
02/0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo
01T0oooo00<000000?ooo`3oool0703oool00`000000oooo0?ooo`2?0?ooo`030000003oool0oooo
0080oooo000[0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool05P3oool00`000000
oooo0?ooo`0I0?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool0S`3oool00`000000
oooo0?ooo`020?ooo`00:`3oool2000000P0oooo00<000000?ooo`3oool05P3oool00`000000oooo
0?ooo`0I0?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool0S`3oool2000000@0oooo
000[0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?oo
o`0J0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0T03oool00`000000oooo0?oo
o`020?ooo`00:`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01D0oooo00<00000
0?ooo`3oool06`3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo0940oooo00<00000
0?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0E0?oo
o`030000003oool0oooo01/0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`2A0?oo
o`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3o
ool05@3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3o
ool0TP3oool00`000000oooo0?ooo`020?ooo`001@3oool2000000<0oooo00D000000?ooo`3oool0
oooo000000060?ooo`@000000`3oool2000000d0oooo00<000000?ooo`3oool01`3oool00`000000
oooo0?ooo`0E0?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool05`3oool00`000000
oooo0?ooo`2B0?ooo`030000003oool0oooo0080oooo00040?ooo`040000003oool0oooo000000<0
oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool0703oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo09<0oooo00<000000?oo
o`3oool00P3oool000@0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`080?oo
o`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0
oooo00L0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0L0?ooo`030000003oool0
oooo01H0oooo00<000000?ooo`3oool0T`3oool00`000000oooo0?ooo`020?ooo`00103oool30000
00@0oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool000000080oooo00<00000
0?ooo`3oool02P3oool3000000L0oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`0N
0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0T`3oool2000000@0oooo00040?oo
o`030000003oool0oooo00<0oooo00D000000?ooo`3oool0oooo000000080?ooo`050000003oool0
oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00L0oooo00<0
00000?ooo`3oool0503oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo01<0oooo00<0
00000?ooo`3oool0U@3oool00`000000oooo0?ooo`020?ooo`00103oool00`000000oooo0?ooo`0>
0?ooo`<000000`3oool010000000oooo0?ooo`0000070?ooo`<000000P3oool3000000L0oooo00<0
00000?ooo`3oool0503oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo01<0oooo00<0
00000?ooo`3oool0U@3oool00`000000oooo0?ooo`020?ooo`001@3oool3000000l0oooo00<00000
0?ooo`3oool00P3oool2000000T0oooo00@000000?ooo`3oool0oooo0P0000080?ooo`030000003o
ool0oooo01@0oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`0A0?ooo`030000003o
ool0oooo09H0oooo00<000000?ooo`3oool00P3oool00240oooo0`0000030?ooo`040000003oool0
oooo0?ooo`800000203oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo01l0oooo00<0
00000?ooo`3oool0403oool00`000000oooo0?ooo`2G0?ooo`030000003oool0oooo0080oooo000W
0?ooo`040000003oool0oooo0?ooo`800000203oool00`000000oooo0?ooo`0D0?ooo`030000003o
ool0oooo01l0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`2G0?ooo`030000003o
ool0oooo0080oooo000V0?ooo`8000000`3oool2000000P0oooo00<000000?ooo`3oool04`3oool0
0`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0V03oool0
0`000000oooo0?ooo`020?ooo`009`3oool010000000oooo0?ooo`3oool2000000P0oooo00<00000
0?ooo`3oool04`3oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo00d0oooo00<00000
0?ooo`3oool0V@3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo01<0oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`0<0?oo
o`030000003oool0oooo09X0oooo00<000000?ooo`3oool00P3oool002/0oooo0P0000080?ooo`03
0000003oool0oooo01<0oooo00<000000?ooo`3oool08P3oool00`000000oooo0?ooo`0;0?ooo`03
0000003oool0oooo09T0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`080?ooo`030000
003oool0oooo0180oooo00<000000?ooo`3oool08P3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo09/0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0203o
ool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo02<0oooo00<000000?ooo`3oool02@3o
ool00`000000oooo0?ooo`2L0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0
oooo00P0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0T0?ooo`030000003oool0
oooo00H0oooo0P00002O0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo
00P0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0T0?ooo`8000001P3oool00`00
0000oooo0?ooo`2O0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00P0
oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0V0?ooo`<000000P3oool00`000000
oooo0?ooo`2P0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00P0oooo
00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0Y0?ooo`800000X`3oool00`000000oooo
0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo0140oooo00<0
00000?ooo`3oool0cP3oool00`000000oooo0?ooo`020?ooo`00:`3oool2000000T0oooo00<00000
0?ooo`3oool0403oool00`000000oooo0?ooo`3>0?ooo`800000103oool002/0oooo00<000000?oo
o`3oool0203oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0<l0oooo00<000000?oo
o`3oool00P3oool002/0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0@0?ooo`03
0000003oool0oooo0<l0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0
203oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0<l0oooo00<000000?ooo`3oool0
0P3oool002/0oooo00<000000?ooo`3oool0203oool00`000000oooo00000002000000h0oooo00<0
00000?ooo`3oool0c`3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`08
0?ooo`8000000P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo0=00oooo00<00000
0?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo0=00oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3o
ool03@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0=00oooo00<000000?ooo`3o
ool00P3oool002/0oooo0P00000>0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool0
c`3oool2000000@0oooo000[0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`3@0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`3@0?ooo`030000003oool0
oooo0080oooo000[0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool02@3oool00`00
0000oooo0?ooo`3A0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`3A0?ooo`030000003oool0oooo0080
oooo00050?ooo`8000000`3oool01@000000oooo0?ooo`3oool0000000H0oooo100000030?ooo`80
00003@3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool0d@3oool00`000000oooo0?ooo`020?ooo`00103oool010000000oooo0?ooo`0000030?ooo`03
0000003oool0000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00X0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo
0=40oooo00<000000?ooo`3oool00P3oool000@0oooo00@000000?ooo`3oool00000103oool00`00
0000oooo0?ooo`080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`3B
0?ooo`030000003oool0oooo0080oooo00050?ooo`800000103oool00`000000oooo000000090?oo
o`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0
oooo00h0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`3B0?ooo`030000003oool0
oooo0080oooo00040?ooo`040000003oool0oooo00000080oooo00D000000?ooo`3oool0oooo0000
00080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`<00000
3`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo0=80oooo0P0000040?ooo`00103o
ool010000000oooo0?ooo`00000=0?ooo`<000000`3oool010000000oooo0?ooo`00000<0?ooo`03
0000003oool0oooo00l0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`3C0?ooo`03
0000003oool0oooo0080oooo00050?ooo`800000403oool00`000000oooo0?ooo`020?ooo`800000
203oool300000080oooo0`00000?0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
e03oool00`000000oooo0?ooo`020?ooo`009`3oool010000000oooo0?ooo`3oool200000100oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`3D0?ooo`030000003oool0oooo0080oooo
000Q0?ooo`<000000`3oool010000000oooo0?ooo`3oool200000100oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`3E0?ooo`030000003oool0oooo0080oooo000W0?ooo`040000003o
ool0oooo0?ooo`8000004@3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0=D0oooo
00<000000?ooo`3oool00P3oool002H0oooo0P0000030?ooo`8000004@3oool00`000000oooo0000
000200000=P0oooo00<000000?ooo`3oool00P3oool002L0oooo00@000000?ooo`3oool0oooo0P00
000A0?ooo`800000f`3oool00`000000oooo0?ooo`020?ooo`00:`3oool200000>d0oooo0P000004
0?ooo`00:`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo02/0oooo00<000000?oo
o`3oool0:P3oool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo02/0oooo00<000000?oo
o`3oool0:`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo000[0?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo0080oooo000[0?oooo400000103oool00?l0oooo8@3oool00?l0oooo8@3oool0
0?l0oooo8@3oool00?l0oooo8@3oool00=d0oooo00<000000?ooo`3oool0@03oool00=d0oooo0P00
00110?ooo`00D`3oool3000004L0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0]
0?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool0oooo0P0000110?ooo`00EP3oool0
0`000000oooo0?ooo`140?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0;@3oool0
0`000000oooo0?ooo`030?ooo`040000003oool0oooo0?ooo`<00000@03oool003H0oooo0P000003
0?ooo`@00000103oool2000000050?ooo`00000000000000003oool00P0000020?ooo`800000103o
ool3000000<0oooo0P0000020?ooo`<0000000<0oooo0000000000000P3oool2000000030?ooo`00
0000oooo00D00000203oool300000080oooo100000020?ooo`@000000`3oool2000000<0oooo1000
000:0?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool0oooo0`0000090?ooo`<00000
503oool5000000@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0?0?ooo`040000
003oool0oooo0?ooo`<000002`3oool200000080oooo1P0000000`3oool000000?ooo`0200000080
oooo0`00000?0?ooo`00=@3oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00<0
oooo00H000000?ooo`3oool000000?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo00L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0
oooo0080oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`0:0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`090?ooo`030000003o
ool0oooo01D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo00h0oooo00<000000?ooo`0000001@3oool00`000000oooo0?ooo`070?ooo`040000003o
ool0oooo000000<0oooo00P000000?ooo`3oool000000?ooo`000000oooo00000080oooo00<00000
0?ooo`3oool0403oool003D0oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`03
0?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool01000
0000oooo0?ooo`0000030?ooo`<00000103oool00`000000oooo0?ooo`070?ooo`<00000103oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool000000`3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00P0oooo100000090?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0>0?ooo`030000003o
ool0000000<0oooo0P00000:0?ooo`030000003oool0oooo00@0oooo00P000000?ooo`3oool00000
0?ooo`000000oooo00000080oooo0`00000@0?ooo`00=@3oool010000000oooo0?ooo`0000030?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool00P3oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo
0?ooo`0000020?ooo`040000003oool0oooo000000D0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`070?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`020?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool02P3o
ool00`000000oooo0?ooo`030?ooo`070000003oool0oooo0?ooo`000000oooo000000030?ooo`<0
00000`3oool00`000000oooo0000000F0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0=0?ooo`070000003oool0oooo0?ooo`000000oooo0000000<
0?ooo`030000003oool0oooo00@0oooo00P000000?ooo`3oool000000?ooo`000000oooo00000080
oooo00@000000?ooo`3oool000003`3oool003D0oooo00@000000?ooo`3oool000000P3oool00`00
0000oooo00000002000000030?ooo`0000000000008000000P3oool3000000@0oooo0P0000040?oo
o`80000000@0oooo0000003oool0oooo0P0000020?ooo`040000003oool00000000000@0oooo0P00
00050?ooo`030000003oool0oooo00P0oooo0P0000030?ooo`030000003oool000000080000000<0
oooo0000003oool00`0000020?ooo`8000000`3oool00`000000oooo00000002000000T0oooo00<0
00000?ooo`3oool00`3oool010000000oooo0?ooo`0000030?ooo`050000003oool0oooo0?ooo`00
00000`3oool01@000000oooo0?ooo`3oool0000001D0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo00`0oooo0P0000020?ooo`8000000P3oool30000
00P0oooo100000030?ooo`8000000P3oool2000000030?ooo`000000oooo00<0oooo0P00000@0?oo
o`00=P3oool2000000X0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0S0?ooo`03
0000003oool0oooo02d0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`020000000:
0?ooo`00000000000?ooo`000000oooo0000003oool000000?ooo`80000000<0oooo000000000000
4`3oool5000000@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0P0?ooo`030000
003oool0oooo0240oooo00120?ooo`040000003oool0oooo0?ooo`800000903oool2000003h0oooo
1@00001;0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool06P3oool00?l0oooo8@3o
ool00001\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {452.938, 276}} -> {-2.71491, \
1.56077, 0.0218596, 5.99231}}]
}, Open  ]],

Cell["\<\
This computes the solution using an orthogonal projection method \
with an explicit Runge-Kutta method as the basic numerical integration \
scheme.\
\>", "Text"],

Cell[BoxData[{
    \(solop = 
      NDSolve[eqs, vars, time, 
        Method \[Rule] {"\<OrthogonalProjection\>", \[IndentingNewLine]Method \
\[Rule] "\<ExplicitRungeKutta\>", \ 
            Dimensions \[Rule] 
              Dimensions[Y]}]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(solop = \ Y\  /. \ First[solop]; \)}], "Input",
  CellLabel->"In[40]:="],

Cell["\<\
The componentwise absolute global error at the end of the \
integration interval is roughly the same as before since the absolute and \
relative tolerances used in the numerical integration are the same.\
\>", \
"Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((solexact\  - \ solop)\)\  /. \ t \[Rule] tfinal\)], "Input",
  CellLabel->"In[42]:="],

Cell[BoxData[
    \({{\(-2.0340697713411987`*^-11\)}, {2.55351295663786`*^-15}}\)], "Output",\

  CellLabel->"Out[42]="]
}, Open  ]],

Cell["\<\
Using the orthogonal projection method however, the deviation from \
the Stiefel manifold is reduced to the level of roundoff.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(OrthogonalErrorPlot[solop]; \)], "Input",
  CellLabel->"In[43]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.190476 0.0147151 1.32542e+15 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.21429 -0.0125 -3 -9 ]
[.21429 -0.0125 3 0 ]
[.40476 -0.0125 -3 -9 ]
[.40476 -0.0125 3 0 ]
[.59524 -0.0125 -3 -9 ]
[.59524 -0.0125 3 0 ]
[.78571 -0.0125 -3 -9 ]
[.78571 -0.0125 3 0 ]
[.97619 -0.0125 -3 -9 ]
[.97619 -0.0125 3 0 ]
[ 0 0 -0.125 0 ]
[-0.0125 .01472 -6 -4.5 ]
[-0.0125 .01472 0 4.5 ]
[-0.0125 .14726 -37.5 -6.25 ]
[-0.0125 .14726 0 6.25 ]
[-0.0125 .2798 -37.5 -6.25 ]
[-0.0125 .2798 0 6.25 ]
[-0.0125 .41234 -37.5 -6.25 ]
[-0.0125 .41234 0 6.25 ]
[-0.0125 .54488 -37.5 -6.25 ]
[-0.0125 .54488 0 6.25 ]
[ 0 0 -0.125 0 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[.5 .63053 -113.125 0 ]
[.5 .63053 113.125 13.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.21429 0 m
.21429 .00625 L
s
[(1)] .21429 -0.0125 0 1 Mshowa
.40476 0 m
.40476 .00625 L
s
[(2)] .40476 -0.0125 0 1 Mshowa
.59524 0 m
.59524 .00625 L
s
[(3)] .59524 -0.0125 0 1 Mshowa
.78571 0 m
.78571 .00625 L
s
[(4)] .78571 -0.0125 0 1 Mshowa
.97619 0 m
.97619 .00625 L
s
[(5)] .97619 -0.0125 0 1 Mshowa
.125 Mabswid
.0619 0 m
.0619 .00375 L
s
.1 0 m
.1 .00375 L
s
.1381 0 m
.1381 .00375 L
s
.17619 0 m
.17619 .00375 L
s
.25238 0 m
.25238 .00375 L
s
.29048 0 m
.29048 .00375 L
s
.32857 0 m
.32857 .00375 L
s
.36667 0 m
.36667 .00375 L
s
.44286 0 m
.44286 .00375 L
s
.48095 0 m
.48095 .00375 L
s
.51905 0 m
.51905 .00375 L
s
.55714 0 m
.55714 .00375 L
s
.63333 0 m
.63333 .00375 L
s
.67143 0 m
.67143 .00375 L
s
.70952 0 m
.70952 .00375 L
s
.74762 0 m
.74762 .00375 L
s
.82381 0 m
.82381 .00375 L
s
.8619 0 m
.8619 .00375 L
s
.9 0 m
.9 .00375 L
s
.9381 0 m
.9381 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
0 .01472 m
.00625 .01472 L
s
[(0)] -0.0125 .01472 1 0 Mshowa
0 .14726 m
.00625 .14726 L
s
gsave
-0.0125 .14726 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(1) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
0 .2798 m
.00625 .2798 L
s
gsave
-0.0125 .2798 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(2) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
0 .41234 m
.00625 .41234 L
s
gsave
-0.0125 .41234 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(3) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
0 .54488 m
.00625 .54488 L
s
gsave
-0.0125 .54488 -37.5 -4.25 Mabsadd m
1 1 Mabs scale
/Courier findfont 10 scalefont setfont
(4) show
/Math1Mono findfont 10 scalefont setfont
(\\264) show
/Courier findfont 10 scalefont setfont
(10) show
0 5.25 rmoveto
/Courier findfont 7.5 scalefont setfont
(-16) show
grestore
.125 Mabswid
0 .04122 m
.00375 .04122 L
s
0 .06773 m
.00375 .06773 L
s
0 .09424 m
.00375 .09424 L
s
0 .12075 m
.00375 .12075 L
s
0 .17377 m
.00375 .17377 L
s
0 .20027 m
.00375 .20027 L
s
0 .22678 m
.00375 .22678 L
s
0 .25329 m
.00375 .25329 L
s
0 .30631 m
.00375 .30631 L
s
0 .33282 m
.00375 .33282 L
s
0 .35932 m
.00375 .35932 L
s
0 .38583 m
.00375 .38583 L
s
0 .43885 m
.00375 .43885 L
s
0 .46536 m
.00375 .46536 L
s
0 .49187 m
.00375 .49187 L
s
0 .51837 m
.00375 .51837 L
s
0 .57139 m
.00375 .57139 L
s
0 .5979 m
.00375 .5979 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
.02381 .61178 m
.02381 .61803 L
s
.21429 .61178 m
.21429 .61803 L
s
.40476 .61178 m
.40476 .61803 L
s
.59524 .61178 m
.59524 .61803 L
s
.78571 .61178 m
.78571 .61803 L
s
.97619 .61178 m
.97619 .61803 L
s
.125 Mabswid
.0619 .61428 m
.0619 .61803 L
s
.1 .61428 m
.1 .61803 L
s
.1381 .61428 m
.1381 .61803 L
s
.17619 .61428 m
.17619 .61803 L
s
.25238 .61428 m
.25238 .61803 L
s
.29048 .61428 m
.29048 .61803 L
s
.32857 .61428 m
.32857 .61803 L
s
.36667 .61428 m
.36667 .61803 L
s
.44286 .61428 m
.44286 .61803 L
s
.48095 .61428 m
.48095 .61803 L
s
.51905 .61428 m
.51905 .61803 L
s
.55714 .61428 m
.55714 .61803 L
s
.63333 .61428 m
.63333 .61803 L
s
.67143 .61428 m
.67143 .61803 L
s
.70952 .61428 m
.70952 .61803 L
s
.74762 .61428 m
.74762 .61803 L
s
.82381 .61428 m
.82381 .61803 L
s
.8619 .61428 m
.8619 .61803 L
s
.9 .61428 m
.9 .61803 L
s
.9381 .61428 m
.9381 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 .01472 m
1 .01472 L
s
.99375 .14726 m
1 .14726 L
s
.99375 .2798 m
1 .2798 L
s
.99375 .41234 m
1 .41234 L
s
.99375 .54488 m
1 .54488 L
s
.125 Mabswid
.99625 .04122 m
1 .04122 L
s
.99625 .06773 m
1 .06773 L
s
.99625 .09424 m
1 .09424 L
s
.99625 .12075 m
1 .12075 L
s
.99625 .17377 m
1 .17377 L
s
.99625 .20027 m
1 .20027 L
s
.99625 .22678 m
1 .22678 L
s
.99625 .25329 m
1 .25329 L
s
.99625 .30631 m
1 .30631 L
s
.99625 .33282 m
1 .33282 L
s
.99625 .35932 m
1 .35932 L
s
.99625 .38583 m
1 .38583 L
s
.99625 .43885 m
1 .43885 L
s
.99625 .46536 m
1 .46536 L
s
.99625 .49187 m
1 .49187 L
s
.99625 .51837 m
1 .51837 L
s
.99625 .57139 m
1 .57139 L
s
.99625 .5979 m
1 .5979 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
gsave
.5 .63053 -174.125 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 21.5 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 12.812 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Orthogonal) show
129.000 12.812 moveto
(error) show
165.000 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
177.000 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
183.000 9.000 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(T) show
189.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(Y) show
201.125 12.812 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
213.125 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(I) show
219.125 12.812 moveto
%%IncludeResource: font Mathematica2Mono
%%IncludeFont: Mathematica2Mono
/Mathematica2Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\310\\310) show
231.125 16.500 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(F) show
243.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(vs) show
261.250 12.812 moveto
(time) show
285.250 12.812 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.02381 .30902 m
.04624 .01472 L
.06032 .30902 L
.07911 .30902 L
.0965 .01472 L
.11523 .16187 L
.13448 .01472 L
.15391 .01472 L
.17399 .45617 L
.19496 .30902 L
.21736 .30902 L
.24074 .16187 L
.26499 .30902 L
.29042 .01472 L
.31618 .01472 L
.34252 .01472 L
.36945 .45617 L
.3971 .30902 L
.42553 .30902 L
.45482 .01472 L
.48505 .16187 L
.51632 .30902 L
.54871 .30902 L
.58233 .01472 L
.61728 .01472 L
.65369 .60332 L
.69169 .01472 L
.73141 .16187 L
.77304 .30902 L
.81674 .16187 L
.86274 .30902 L
.91127 .30902 L
.94373 .01472 L
.97619 .01472 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[43]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00<03oool2000002/0oooo1000000Y0?ooo`@00000:`3oool2000002d0
oooo00<000000?ooo`3oool0:@3oool3000000T0oooo000_0?ooo`040000003oool0oooo000002`0
oooo00<000000?ooo`3oool0:03oool00`000000oooo0?ooo`0[0?ooo`040000003oool0oooo0000
02`0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`060?ooo`00;`3oool010000000
oooo0?ooo`00000/0?ooo`030000003oool0oooo02T0oooo00<000000?ooo`3oool0;@3oool00`00
0000oooo0?ooo`0W0?ooo`D00000;@3oool00`000000oooo0?ooo`060?ooo`00;`3oool010000000
oooo0?ooo`00000/0?ooo`030000003oool0oooo02X0oooo00<000000?ooo`3oool0:P3oool20000
02X0oooo00@000000?ooo`3oool00000:`3oool3000000T0oooo000_0?ooo`040000003oool0oooo
000002`0oooo00<000000?ooo`3oool0:`3oool00`000000oooo0?ooo`0[0?ooo`030000003oool0
oooo02P0oooo00<000000?ooo`000000:`3oool00`000000oooo0?ooo`090?ooo`00;`3oool01000
0000oooo0?ooo`00000Z0?ooo`<00000:P3oool010000000oooo0?ooo`00000Z0?ooo`040000003o
ool0oooo000002/0oooo0P00000[0?ooo`030000003oool0oooo00T0oooo000`0?ooo`800000;@3o
ool00`000000oooo0?ooo`0Y0?ooo`800000;03oool2000002d0oooo00<000000?ooo`3oool0:@3o
ool4000000P0oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo000[0?oooo400000
103oool002@0oooo0P0000050?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0:`3o
ool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo02/0oooo00<000000?ooo`3oool0:`3o
ool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3o
ool002<0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`3]0?ooo`030000003o
ool0oooo0080oooo000S0?ooo`040000003oool0oooo000000@0oooo0`0000080?ooo`030000003o
ool0oooo00T0oooo00<000000?ooo`3oool01P3oool600000200oooo3@00000J0?ooo`030000003o
ool0oooo01`0oooo2@00000A0?ooo`030000003oool0oooo03X0oooo2@0000030?ooo`800000103o
ool002<0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`080?ooo`030000003o
ool0oooo00T0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo01h0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0H0?ooo`030000003o
ool0oooo01`0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0?0?ooo`030000003o
ool0oooo03X0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`020?ooo`008`3oool0
10000000oooo0?ooo`0000040?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool02@3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool07P3o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0703o
ool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00l0oooo0P00000k0?ooo`030000003o
ool0oooo00X0oooo00<000000?ooo`3oool00P3oool002<0oooo00@000000?ooo`3oool00000103o
ool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool02@3o
ool00`000000oooo0?ooo`0H0?ooo`8000007@3oool00`000000oooo0?ooo`050?ooo`030000003o
ool0oooo00l0oooo0P00000k0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool00P3o
ool002@0oooo0P0000050?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool02@3oool2
000000L0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0
oooo00T0oooo00<000000?ooo`3oool0603oool2000001d0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`0?0?ooo`800000>`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0
oooo0080oooo000[0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool02@3oool20000
00H0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo
00T0oooo00<000000?ooo`3oool0603oool2000001`0oooo00<000000?ooo`3oool01P3oool00`00
0000oooo0?ooo`0?0?ooo`800000>P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo
0080oooo000[0?ooo`8000002@3oool00`000000oooo0?ooo`090?ooo`8000001P3oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3oool02P3oool00`000000
oooo0?ooo`0G0?ooo`030000003oool0000001`0oooo00<000000?ooo`3oool01P3oool00`000000
oooo0?ooo`0?0?ooo`030000003oool0000003T0oooo00<000000?ooo`3oool02P3oool2000000@0
oooo000[0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool02@3oool2000000H0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo00X0oooo
00<000000?ooo`3oool05`3oool010000000oooo0?ooo`00000K0?ooo`030000003oool0oooo00H0
oooo00<000000?ooo`3oool03`3oool00`000000oooo0000000i0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01`3oool2000000/0oooo
0P0000060?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool07@3oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo01H0oooo00@000000?ooo`3oool000006`3oool00`000000
oooo0?ooo`070?ooo`030000003oool0oooo00d0oooo00@000000?ooo`3oool00000>@3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`0000002@3oool00`000000oooo000000060?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool07@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo01H0
oooo00@000000?ooo`3oool000006`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo
00d0oooo00D000000?ooo`3oool0oooo0000000h0?ooo`030000003oool0oooo00/0oooo00<00000
0?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01`3oool00`000000oooo000000090?oo
o`030000003oool0000000H0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0M0?oo
o`030000003oool0oooo00/0oooo00<000000?ooo`3oool05P3oool010000000oooo0?ooo`00000K
0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool03@3oool01@000000oooo0?ooo`3o
ool0000003P0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`020?ooo`00:`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0000000T0oooo00@000000?ooo`3oool000001@3o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool02`3o
ool00`000000oooo0?ooo`0F0?ooo`050000003oool0oooo0?ooo`0000006P3oool00`000000oooo
0?ooo`070?ooo`030000003oool0oooo00d0oooo00D000000?ooo`3oool0oooo0000000h0?ooo`03
0000003oool0oooo00/0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0
1`3oool00`000000oooo000000090?ooo`040000003oool0oooo000000D0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`0L0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool05P3oool01@000000oooo0?ooo`3oool0000001X0oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`0=0?ooo`050000003oool0oooo0?ooo`000000>03oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo0080oooo000[0?ooo`800000203oool00`000000oooo000000090?oo
o`040000003oool0oooo000000D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0L
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool05P3oool01@000000oooo0?ooo`3o
ool0000001X0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0=0?ooo`030000003o
ool0oooo0080oooo00<000000?ooo`3oool0=@3oool00`000000oooo0?ooo`0:0?ooo`800000103o
ool002/0oooo00<000000?ooo`3oool01`3oool00`000000oooo000000090?ooo`040000003oool0
oooo000000@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0L0?ooo`030000003o
ool0oooo00/0oooo00<000000?ooo`3oool05P3oool01@000000oooo0?ooo`3oool0000001T0oooo
00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo0080oooo
00<000000?ooo`3oool0=03oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo0080oooo
000[0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`0000002@3oool010000000oooo0?oo
o`0000040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0703oool00`000000oooo
0?ooo`0;0?ooo`030000003oool0oooo01H0oooo00D000000?ooo`3oool0oooo0000000I0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo03@0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`020?ooo`00
:`3oool00`000000oooo0?ooo`070?ooo`030000003oool0000000T0oooo00@000000?ooo`3oool0
0000103oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool05P3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0c0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3o
ool00P3oool002/0oooo00<000000?ooo`3oool01`3oool00`000000oooo000000090?ooo`040000
003oool0oooo000000@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0L0?ooo`03
0000003oool0oooo00/0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo01H0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool0<`3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`000000
2@3oool01@000000oooo0?ooo`3oool0000000<0oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`0K0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool05@3oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0203oool00`000000
oooo0?ooo`0=0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0<`3oool00`000000
oooo0?ooo`0<0?ooo`030000003oool0oooo0080oooo000[0?ooo`800000203oool00`000000oooo
000000090?ooo`050000003oool0oooo0?ooo`0000000`3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo01/0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0E0?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo00d0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0c0?ooo`03
0000003oool0oooo00/0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`070?ooo`030000
003oool0000000T0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo01D0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo00P0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo0380
oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo
0?ooo`070?ooo`030000003oool0000000T0oooo00D000000?ooo`3oool0oooo000000030?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo01D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0E0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo0380oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`020?ooo`00
:`3oool00`000000oooo0?ooo`060?ooo`040000003oool0oooo000000T0oooo00D000000?ooo`3o
ool0oooo000000020?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool06`3oool00`00
0000oooo0?ooo`0<0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`0E0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool03@3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo0380oooo00<000000?ooo`3oool0303oool00`00
0000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`060?ooo`040000003oool0oooo0000
00T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00H0oooo00<00000
0?ooo`3oool06`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo01@0oooo00<00000
0?ooo`3oool0103oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool03@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo0300oooo00<00000
0?ooo`3oool03@3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`060?oo
o`040000003oool0oooo000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003o
ool0oooo00H0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`0=0?ooo`030000003o
ool0oooo01@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0D0?ooo`030000003o
ool0oooo00X0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo0300oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`020?ooo`00:`3oool0
0`000000oooo0?ooo`060?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo000000080?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04`3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`0`0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool00P3o
ool002/0oooo0P0000070?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo000000080?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool04`3o
ool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`0`0?ooo`030000003oool0oooo00`0oooo0P0000040?ooo`00:`3oool0
0`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`0000001`3oool00`000000oooo0?oo
o`030?ooo`030000003oool0000000P0oooo00<000000?ooo`3oool06`3oool00`000000oooo0?oo
o`0=0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?oo
o`0C0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?oo
o`070?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?oo
o`020?ooo`00:`3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`0000001`3o
ool00`000000oooo0?ooo`030?ooo`030000003oool0000000P0oooo00<000000?ooo`3oool06`3o
ool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02`3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`020?ooo`00103oool400000080oooo00D000000?ooo`3oool0oooo0000
00060?ooo`@000000`3oool2000000d0oooo00<000000?ooo`3oool01P3oool01@000000oooo0?oo
o`3oool0000000L0oooo00<000000?ooo`3oool00`3oool00`000000oooo000000080?ooo`030000
003oool0oooo01/0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0C0?ooo`030000
003oool0oooo00L0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0_0?ooo`030000
003oool0oooo00d0oooo00<000000?ooo`3oool00P3oool000H0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo000000090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo
0?ooo`0:0?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo000000070?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`000000203oool00`000000oooo0?ooo`0J0?ooo`03
0000003oool0oooo00h0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo0180oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0;0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool0;P3oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo0080oooo00060?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
203oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool00`000000
oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`0000001`3oool00`000000oooo0?ooo`030?oo
o`8000002@3oool00`000000oooo0?ooo`0J0?ooo`030000003oool0oooo00h0oooo00<000000?oo
o`3oool04`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0180oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool0;P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo0080oooo00060?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool00000
0080oooo00<000000?ooo`3oool02P3oool3000000H0oooo00D000000?ooo`3oool0oooo00000007
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0I
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`0=
0?ooo`800000103oool000H0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0
oooo00H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo00<0
00000?ooo`3oool01P3oool01@000000oooo0?ooo`3oool0000000L0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0
3P3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0
403oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0
203oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0
0P3oool000@0oooo0`00000>0?ooo`<000000`3oool010000000oooo0?ooo`0000070?ooo`<00000
0P3oool2000000L0oooo00D000000?ooo`3oool0oooo000000070?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00T0
oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo0080
oooo00060?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool00P3oool2000000T0oooo
00L000000?ooo`3oool0oooo0000003oool0000000H0oooo00D000000?ooo`3oool0oooo00000007
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0C
0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0A
0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`02
0?ooo`008@3oool3000000<0oooo00@000000?ooo`3oool0oooo0`0000050?ooo`030000003oool0
oooo0080oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0
oooo0180oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0
oooo00/0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`090?ooo`030000003oool0
oooo0140oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0>0?ooo`030000003oool0
oooo0080oooo000W0?ooo`050000003oool0oooo0?ooo`0000001`3oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0B
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0?
0?ooo`030000003oool0000001T0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`02
0?ooo`009P3oool2000000<0oooo0P0000060?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool01@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo0100oooo00<000000?oo
o`3oool01P3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo0140oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00/0oooo00<000000?oo
o`3oool02`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00l0oooo00<000000?oo
o`0000006@3oool00`000000oooo0?ooo`0=0?ooo`800000103oool002L0oooo00@000000?ooo`3o
ool0oooo0`0000050?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`0000001`3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool02`3oool00`00
0000oooo0?ooo`0:0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`0000006@3oool00`00
0000oooo0?ooo`0>0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00l0
oooo00<000000?ooo`3oool03`3oool00`000000oooo000000070?ooo`030000003oool0oooo00l0
oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool03`3oool010000000oooo0?ooo`00000G0?ooo`030000003oool0oooo
00l0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool03`3oool00`000000
oooo0?ooo`0?0?ooo`030000003oool0000000L0oooo00<000000?ooo`3oool03`3oool00`000000
oooo0?ooo`0A0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool03P3oool00`000000
oooo0?ooo`0=0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0303oool00`000000
oooo0?ooo`0=0?ooo`050000003oool0oooo0?ooo`0000005`3oool00`000000oooo0?ooo`0?0?oo
o`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3o
ool03`3oool00`000000oooo000000060?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool04@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3o
ool03@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3o
ool03@3oool01@000000oooo0?ooo`3oool0000001L0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo0080
oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00l0
oooo00<000000?ooo`0000001P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0140
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00d0
oooo00D000000?ooo`3oool0oooo0000000G0?ooo`030000003oool0oooo00l0oooo00<000000?oo
o`3oool00P3oool002/0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0?0?ooo`03
0000003oool0000000H0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0A0?ooo`03
0000003oool0oooo00/0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo00T0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool0503oool00`000000oooo0?ooo`0?0?ooo`03
0000003oool0oooo0080oooo000[0?ooo`8000001P3oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo00<0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0>0?ooo`050000
003oool0oooo0?ooo`0000001@3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0140
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00h0
oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00h0oooo
00D000000?ooo`3oool0oooo000000050?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool0403oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3o
ool03@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3o
ool02`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3o
ool03`3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo00<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0@0?ooo`030000
003oool0oooo00h0oooo00D000000?ooo`3oool0oooo000000050?ooo`030000003oool0oooo0100
oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00d0
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01<0
oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`0@0?ooo`030000003oool0oooo00h0oooo00D000000?ooo`3oool0oooo000000050?ooo`03
0000003oool0oooo0100oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0=0?ooo`03
0000003oool0oooo00`0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo00h0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo01<0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`020?ooo`00
:`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00h0oooo00D000000?ooo`3oool0
oooo000000050?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0403oool00`000000
oooo0?ooo`0=0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool03P3oool00`000000
oooo0?ooo`090?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool02@3oool00`000000
oooo0?ooo`050?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool0403oool00`000000
oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00h0oooo
00D000000?ooo`3oool0oooo000000050?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool0403oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool03P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3o
ool02@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo0140oooo00<000000?ooo`3o
ool0403oool00`000000oooo0?ooo`020?ooo`00:`3oool2000000D0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0
3@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
403oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool0
2`3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
3P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0
4@3oool00`000000oooo0?ooo`0?0?ooo`800000103oool002/0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0403o
ool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool02@3o
ool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01`3o
ool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool00P3o
ool002/0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo00h0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0
oooo00L0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0
oooo0100oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool04@3oool00`00
0000oooo0?ooo`0<0?ooo`030000003oool0oooo00<0oooo00D000000?ooo`3oool0oooo0000000D
0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0?
0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?oo
o`3oool04@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00<0oooo00D000000?oo
o`3oool0oooo0000000D0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool03P3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool01`3oool0
0`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool02@3oool0
0`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool00P3oool0
02/0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo
00<0oooo00D000000?ooo`3oool0oooo0000000D0?ooo`030000003oool0oooo00h0oooo00<00000
0?ooo`3oool03`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00l0oooo00<00000
0?ooo`3oool01`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0100oooo00<00000
0?ooo`3oool00P3oool002/0oooo0P0000050?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00/0oooo00<000000?oo
o`3oool01@3oool010000000oooo0?ooo`00000D0?ooo`030000003oool0oooo00h0oooo00<00000
0?ooo`3oool03`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00l0oooo00<00000
0?ooo`3oool01`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00l0oooo0P000004
0?ooo`00:`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00/0oooo00<000000?oo
o`3oool01@3oool010000000oooo0?ooo`00000D0?ooo`030000003oool0oooo00h0oooo00<00000
0?ooo`3oool03`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo0100oooo00<00000
0?ooo`3oool01`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`3oool02P3oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo0140oooo00<00000
0?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo00D0oooo00@000000?ooo`3oool00000503oool00`000000oooo0?ooo`0>
0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`03
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0A
0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0140oooo00<000000?oo
o`3oool02`3oool00`000000oooo0?ooo`050?ooo`040000003oool0oooo000001@0oooo00<00000
0?ooo`3oool03P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool0403oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0180oooo00<00000
0?ooo`3oool00`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00d0oooo00<00000
0?ooo`3oool04@3oool00`000000oooo0?ooo`020?ooo`00103oool400000080oooo00D000000?oo
o`3oool0oooo000000060?ooo`@000000`3oool2000000d0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool04@3oool0
0`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00D0oooo00@000000?ooo`3oool00000503o
ool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0203o
ool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool04P3o
ool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo0080oooo00040?ooo`030000003oool0
oooo00@0oooo00<000000?ooo`0000002@3oool01@000000oooo0?ooo`3oool000000080oooo00<0
00000?ooo`3oool02P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00/0oooo00<0
00000?ooo`3oool01@3oool010000000oooo0?ooo`00000D0?ooo`030000003oool0oooo00h0oooo
00<000000?ooo`3oool0403oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo0100oooo
00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo0140oooo
00<000000?ooo`3oool00P3oool000D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?oo
o`080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool01@000000oooo0?ooo`3oool0000001@0
oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`070?ooo`030000003oool0000001@0
oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00P0
oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo01<0
oooo00D000000?ooo`3oool0oooo0000000@0?ooo`030000003oool0oooo00`0oooo00<000000?oo
o`3oool04@3oool00`000000oooo0?ooo`020?ooo`001P3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0000000T0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00X0oooo0`0000030?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo0000000D
0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool01`3oool00`000000oooo0000000D
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0C
0?ooo`050000003oool0oooo0?ooo`000000403oool00`000000oooo0?ooo`0<0?ooo`030000003o
ool0oooo0100oooo0P0000040?ooo`001`3oool010000000oooo0?ooo`0000030?ooo`030000003o
ool0oooo00H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`00
0000503oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`00
0000503oool00`000000oooo0?ooo`0>0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3o
ool04`3oool01@000000oooo0?ooo`3oool000000100oooo00<000000?ooo`3oool0303oool00`00
0000oooo0?ooo`0A0?ooo`030000003oool0oooo0080oooo00040?ooo`040000003oool0oooo0000
00d0oooo0`0000030?ooo`040000003oool0oooo000000L0oooo0`0000020?ooo`800000103oool0
0`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`000000503oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo00L0oooo0P00000E0?ooo`030000003oool0oooo00d0oooo00<0
00000?ooo`3oool04P3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0100oooo00<0
00000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool000000140oooo00<0
00000?ooo`3oool0303oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo0080oooo0005
0?ooo`800000403oool00`000000oooo0?ooo`020?ooo`8000002@3oool01`000000oooo0?ooo`3o
ool000000?ooo`0000000`3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`00
0000503oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00L0oooo0P00000E0?ooo`03
0000003oool0oooo00d0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo0140oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`03
0000003oool000000180oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0B0?ooo`03
0000003oool0oooo0080oooo000Q0?ooo`<000000`3oool010000000oooo0?ooo`3oool3000000<0
oooo00<000000?ooo`3oool01P3oool01@000000oooo0?ooo`3oool0000001@0oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`070?ooo`8000005@3oool00`000000oooo0?ooo`0=0?oo
o`030000003oool0oooo0180oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00L0oooo00<000000?ooo`3oool0503oool00`000000oooo0000000B0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`020?oo
o`009`3oool01@000000oooo0?ooo`3oool0000000D0oooo00<000000?ooo`3oool01P3oool01@00
0000oooo0?ooo`3oool0000001@0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`09
0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0C
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo01@0oooo00<000000?ooo`0000004P3oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool00P3oool002H0oooo0P0000030?oo
o`800000103oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`000000503oool0
0`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0503oool0
0`000000oooo0?ooo`0<0?ooo`030000003oool0oooo01<0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool05@3oool0
0`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool04@3oool2
000000@0oooo000W0?ooo`040000003oool0oooo0?ooo`<000000`3oool00`000000oooo0?ooo`06
0?ooo`D00000503oool00`000000oooo0?ooo`040?ooo`H000002`3oool00`000000oooo0?ooo`0D
0?ooo`030000003oool0oooo00D0oooo2000000E0?ooo`T00000503oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool04P3oool=000001@0oooo00<00000
0?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool09@3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo02D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0g0?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0C03oool00`000000oooo0?ooo`020?oo
o`00:`3oool00`000000oooo0?ooo`0U0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3o
ool09@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo03L0oooo00<000000?ooo`3o
ool01@3oool00`000000oooo0?ooo`1<0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000
003oool0oooo02D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0U0?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool0>03oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo04`0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool09P3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo02H0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`0h0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0C03o
ool00`000000oooo0?ooo`020?ooo`00:`3oool2000002L0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0V0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0>03oool0
0`000000oooo0?ooo`050?ooo`030000003oool0oooo04/0oooo0P0000040?ooo`00:`3oool00`00
0000oooo0?ooo`0V0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool09P3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo03P0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`1<0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo02H0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0V0?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool0>03oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo04`0
oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool09P3oool00`000000oooo
0?ooo`020?ooo`030000003oool0oooo02H0oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`0h0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0C03oool00`000000oooo
0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0V0?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool09P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo03T0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`1<0?ooo`030000003oool0oooo0080oooo000[
0?ooo`030000003oool0oooo02H0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0V
0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0>@3oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo04`0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?oo
o`3oool09P3oool01@000000oooo0?ooo`3oool0000002T0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0i0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0C03oool0
0`000000oooo0?ooo`020?ooo`00:`3oool2000002L0oooo00D000000?ooo`3oool0oooo0000000Y
0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0>@3oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo04/0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`0V0?oo
o`050000003oool0oooo0?ooo`000000:P3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo03T0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`1<0?ooo`030000003oool0
oooo0080oooo000[0?ooo`030000003oool0oooo02H0oooo00D000000?ooo`3oool0oooo0000000Z
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0>@3oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo04`0oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?oo
o`3oool09P3oool01@000000oooo0?ooo`3oool0000002X0oooo00D000000?ooo`3oool0oooo0000
000l0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0C03oool00`000000oooo0?oo
o`020?ooo`00:`3oool00`000000oooo0?ooo`0V0?ooo`050000003oool0oooo0?ooo`000000:P3o
ool01@000000oooo0?ooo`3oool0000003d0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`1=0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo02H0oooo00D0
00000?ooo`3oool0oooo0000000Z0?ooo`050000003oool0oooo0?ooo`000000?@3oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo04d0oooo00<000000?ooo`3oool00P3oool002/0oooo
0P00000W0?ooo`040000003oool0oooo000002/0oooo00D000000?ooo`3oool0oooo0000000m0?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool0C03oool2000000@0oooo000[0?ooo`03
0000003oool0oooo02H0oooo00@000000?ooo`3oool00000:`3oool01@000000oooo0?ooo`3oool0
000003d0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1=0?ooo`030000003oool0
oooo0080oooo000[0?ooo`030000003oool0oooo02H0oooo00@000000?ooo`3oool00000:`3oool0
1@000000oooo0?ooo`3oool0000003d0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?oo
o`1=0?ooo`030000003oool0oooo0080oooo00050?ooo`8000000`3oool01@000000oooo0?ooo`3o
ool0000000H0oooo100000030?ooo`8000003@3oool00`000000oooo0?ooo`0V0?ooo`040000003o
ool0oooo000002/0oooo00@000000?ooo`3oool00000?P3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo04d0oooo00<000000?ooo`3oool00P3oool000@0oooo00@000000?ooo`3oool0
00000`3oool00`000000oooo000000090?ooo`050000003oool0oooo0?ooo`0000000P3oool00`00
0000oooo0?ooo`0:0?ooo`030000003oool0oooo02H0oooo00@000000?ooo`3oool00000:`3oool0
10000000oooo0?ooo`00000n0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0C@3o
ool00`000000oooo0?ooo`020?ooo`001`3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo00P0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00X0oooo00<0
00000?ooo`3oool09P3oool010000000oooo0?ooo`00000[0?ooo`040000003oool0oooo000003h0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1=0?ooo`030000003oool0oooo0080
oooo00050?ooo`800000103oool00`000000oooo000000090?ooo`050000003oool0oooo0?ooo`00
00000P3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo02H0oooo00@000000?ooo`3o
ool00000:`3oool010000000oooo0?ooo`00000n0?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool0C@3oool00`000000oooo0?ooo`020?ooo`001`3oool010000000oooo0?ooo`000003
0?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003o
ool0oooo00X0oooo0`00000W0?ooo`800000;03oool010000000oooo0?ooo`00000n0?ooo`030000
003oool0oooo00<0oooo00<000000?ooo`3oool0C03oool2000000@0oooo00040?ooo`040000003o
ool0oooo000000d0oooo0`0000030?ooo`040000003oool0oooo000000`0oooo00<000000?ooo`3o
ool09`3oool2000002`0oooo00@000000?ooo`3oool00000?P3oool00`000000oooo0?ooo`030?oo
o`030000003oool0oooo04d0oooo00<000000?ooo`3oool00P3oool000D0oooo0P00000@0?ooo`03
0000003oool0oooo0080oooo0P0000080?ooo`<000000P3oool2000002P0oooo0P00000/0?ooo`03
0000003oool0000003l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1=0?ooo`03
0000003oool0oooo0080oooo000W0?ooo`070000003oool0oooo0?ooo`000000oooo0000000W0?oo
o`800000;@3oool2000003l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1=0?oo
o`030000003oool0oooo0080oooo000Q0?ooo`<000000`3oool010000000oooo0?ooo`3oool30000
02L0oooo0P00000]0?ooo`800000?`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
04d0oooo00<000000?ooo`3oool00P3oool002L0oooo00D000000?ooo`3oool0oooo0000000Y0?oo
o`800000;@3oool2000003l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1=0?oo
o`030000003oool0oooo0080oooo000V0?ooo`8000000`3oool2000002P0oooo0P00000]0?ooo`80
0000@03oool01@000000oooo0?ooo`3oool0000004l0oooo0P0000040?ooo`009`3oool010000000
oooo0?ooo`3oool3000002L0oooo00<000000?ooo`3oool0;03oool200000400oooo00D000000?oo
o`3oool0oooo0000001@0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo
02L0oooo00<000000?ooo`3oool0;03oool00`000000oooo0?ooo`0o0?ooo`050000003oool0oooo
0?ooo`000000D03oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`0W0?oo
o`030000003oool0oooo02`0oooo00<000000?ooo`3oool0?`3oool01@000000oooo0?ooo`3oool0
00000500oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool09`3oool00`00
0000oooo0?ooo`0/0?ooo`030000003oool0oooo03l0oooo00D000000?ooo`3oool0oooo0000001@
0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo09P0oooo00D000000?oo
o`3oool0oooo0000001@0?ooo`030000003oool0oooo0080oooo000[0?ooo`800000V@3oool01@00
0000oooo0?ooo`3oool0000004l0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`2H0?oo
o`050000003oool0oooo0?ooo`000000D03oool00`000000oooo0?ooo`020?ooo`00:`3oool00`00
0000oooo0?ooo`2H0?ooo`050000003oool0oooo0?ooo`000000D03oool00`000000oooo0?ooo`02
0?ooo`00:`3oool00`000000oooo0?ooo`2H0?ooo`050000003oool0oooo0?ooo`000000D03oool0
0`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`2H0?ooo`050000003oool0oooo
0?ooo`000000D03oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`2H0?oo
o`050000003oool0oooo0?ooo`000000D03oool00`000000oooo0?ooo`020?ooo`00:`3oool00`00
0000oooo0?ooo`2H0?ooo`050000003oool0oooo0?ooo`000000D03oool00`000000oooo0?ooo`02
0?ooo`00:`3oool2000009T0oooo00D000000?ooo`3oool0oooo0000001?0?ooo`800000103oool0
02/0oooo00<000000?ooo`3oool0V03oool01@000000oooo0?ooo`3oool000000500oooo00<00000
0?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0V03oool01@000000oooo0?ooo`3oool0
00000500oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0V@3oool00`00
0000oooo0000001A0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo09T0
oooo00<000000?ooo`000000D@3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo
0?ooo`2I0?ooo`030000003oool000000540oooo00<000000?ooo`3oool00P3oool002/0oooo0P00
002J0?ooo`030000003oool000000500oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`2I
0?ooo`030000003oool000000540oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?oo
o`3oool0V@3oool00`000000oooo0000001A0?ooo`030000003oool0oooo0080oooo00060?ooo`05
0000003oool0oooo0?ooo`0000000`3oool00`000000oooo0?ooo`040?ooo`@000000`3oool20000
00d0oooo00<000000?ooo`3oool0V@3oool00`000000oooo0000001A0?ooo`030000003oool0oooo
0080oooo00060?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000002@3oool01@000000
oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`2I0?oo
o`030000003oool000000540oooo00<000000?ooo`3oool00P3oool000<0oooo1@0000040?ooo`03
0000003oool0oooo00P0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00X0oooo00<000000?ooo`3oool0V@3oool00`000000oooo0000001A0?ooo`030000003oool0oooo
0080oooo00030?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`0000002@3oool01@00
0000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool02P3oool3000009T0oooo00<00000
0?ooo`000000D03oool2000000@0oooo00040?ooo`030000003oool0000000<0oooo00D000000?oo
o`3oool0oooo000000080?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`0:0?ooo`030000003oool0oooo09T0oooo00<000000?ooo`000000D@3oool00`000000oooo0?oo
o`020?ooo`001@3oool2000000h0oooo0`0000030?ooo`040000003oool0oooo000000L0oooo0`00
00020?ooo`800000VP3oool00`000000oooo0000001A0?ooo`030000003oool0oooo0080oooo0006
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool00P3oool2000000T0oooo00L00000
0?ooo`3oool0oooo0000003oool0000009T0oooo00<000000?ooo`000000D@3oool00`000000oooo
0?ooo`020?ooo`008@3oool3000000<0oooo00@000000?ooo`3oool0oooo0`00002I0?ooo`030000
003oool000000540oooo00<000000?ooo`3oool00P3oool002L0oooo00D000000?ooo`3oool0oooo
0000002K0?ooo`030000003oool000000540oooo00<000000?ooo`3oool00P3oool002H0oooo0P00
00030?ooo`030000003oool0oooo09T0oooo00<000000?ooo`000000D@3oool00`000000oooo0?oo
o`020?ooo`009`3oool010000000oooo0?ooo`3oool3000009X0oooo00<000000?ooo`3oool0C`3o
ool2000000@0oooo000[0?ooo`030000003oool0oooo09X0oooo00<000000?ooo`3oool0D03oool0
0`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`2J0?ooo`030000003oool0oooo
0500oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3oool0VP3oool00`000000
oooo0?ooo`1@0?ooo`030000003oool0oooo0080oooo000[0?ooo`030000003oool0oooo09X0oooo
00<000000?ooo`3oool0D03oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?oo
o`2J0?ooo`030000003oool0oooo0500oooo00<000000?ooo`3oool00P3oool002/0oooo0P00002K
0?ooo`030000003oool0oooo04l0oooo0P0000040?ooo`00:`3oool00`000000oooo0?ooo`2J0?oo
o`030000003oool0oooo0500oooo00<000000?ooo`3oool00P3oool002/0oooo00<000000?ooo`3o
ool0k@3oool00`000000oooo0?ooo`020?ooo`00:`3oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo02/0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`0[0?ooo`030000
003oool0oooo02/0oooo00<000000?ooo`3oool0:`3oool00`000000oooo0?ooo`020?ooo`030000
003oool0oooo0080oooo000[0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01`3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0080oooo000[0?oooo400000103oool0
0?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00=d0oooo00<00000
0?ooo`3oool0@03oool00=d0oooo0P0000110?ooo`00D`3oool3000004L0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0]0?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3o
ool0oooo0P0000110?ooo`00EP3oool00`000000oooo0?ooo`140?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool0;@3oool00`000000oooo0?ooo`030?ooo`040000003oool0oooo0?oo
o`<00000@03oool003H0oooo0P0000030?ooo`@00000103oool2000000050?ooo`00000000000000
003oool00P0000020?ooo`800000103oool3000000<0oooo0P0000020?ooo`<0000000<0oooo0000
000000000P3oool2000000030?ooo`000000oooo00D00000203oool300000080oooo100000020?oo
o`@000000`3oool2000000<0oooo1000000:0?ooo`030000003oool0oooo00<0oooo00@000000?oo
o`3oool0oooo0`0000090?ooo`<00000503oool5000000@0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0?0?ooo`040000003oool0oooo0?ooo`<000002`3oool200000080oooo1P00
00000`3oool000000?ooo`0200000080oooo0`00000?0?ooo`00=@3oool010000000oooo0?ooo`00
00030?ooo`030000003oool0oooo00<0oooo00H000000?ooo`3oool000000?ooo`0000020?ooo`04
0000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool00`000000
oooo0?ooo`020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool000000`3oool00`00
0000oooo0?ooo`0:0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`090?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo00<000000?ooo`0000001@3oool00`00
0000oooo0?ooo`070?ooo`040000003oool0oooo000000<0oooo00P000000?ooo`3oool000000?oo
o`000000oooo00000080oooo00<000000?ooo`3oool0403oool003D0oooo00@000000?ooo`3oool0
00000`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00@000000?ooo`3o
ool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool000000P3oool010000000oooo0?ooo`0000030?ooo`<00000103oool00`000000
oooo0?ooo`070?ooo`<00000103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0080
oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo
00P0oooo100000090?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`0>0?ooo`030000003oool0000000<0oooo0P00000:0?ooo`030000003oool0oooo
00@0oooo00P000000?ooo`3oool000000?ooo`000000oooo00000080oooo0`00000@0?ooo`00=@3o
ool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
0P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000D0oooo
00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`070?ooo`040000003oool0oooo000000<0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`020?ooo`040000003oool0oooo0000
00<0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`030?ooo`070000003oool0oooo
0?ooo`000000oooo000000030?ooo`<000000`3oool00`000000oooo0000000F0?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0=0?ooo`070000003o
ool0oooo0?ooo`000000oooo0000000<0?ooo`030000003oool0oooo00@0oooo00P000000?ooo`3o
ool000000?ooo`000000oooo00000080oooo00@000000?ooo`3oool000003`3oool003D0oooo00@0
00000?ooo`3oool000000P3oool00`000000oooo00000002000000030?ooo`000000000000800000
0P3oool3000000@0oooo0P0000040?ooo`80000000@0oooo0000003oool0oooo0P0000020?ooo`04
0000003oool00000000000@0oooo0P0000050?ooo`030000003oool0oooo00P0oooo0P0000030?oo
o`030000003oool000000080000000<0oooo0000003oool00`0000020?ooo`8000000`3oool00`00
0000oooo00000002000000T0oooo00<000000?ooo`3oool00`3oool010000000oooo0?ooo`000003
0?ooo`050000003oool0oooo0?ooo`0000000`3oool01@000000oooo0?ooo`3oool0000001D0oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00`0oooo
0P0000020?ooo`8000000P3oool3000000P0oooo100000030?ooo`8000000P3oool2000000030?oo
o`000000oooo00<0oooo0P00000@0?ooo`00=P3oool2000000X0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo02d0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`020000000:0?ooo`00000000000?ooo`000000oooo0000003oool00000
0?ooo`80000000<0oooo0000000000004`3oool5000000@0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0P0?ooo`030000003oool0oooo0240oooo00120?ooo`040000003oool0oooo
0?ooo`800000903oool2000003h0oooo1@00001;0?ooo`030000003oool0oooo00@0oooo00<00000
0?ooo`3oool06P3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{74.875, 361.875}, {254.812, 77.875}} -> {-2.71491, \
1.95831, 0.0218596, 3.14144}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Implementation", "Section",
  CellTags->"s:3"],

Cell[TextData[{
  "The implementation of the method ",
  StyleBox["OrthogonalProjection", "MR"],
  " has three basic components:"
}], "Text"],

Cell[TextData[{
  "Initialization. Set up the base method to use in the integration, \
determining any method coefficients and setting up any workspaces that should \
be used. This is done once, before any actual integration is carried out, and \
the resulting ",
  StyleBox["MethodData", "MR"],
  " object is validated so that it does not need to be checked at each \
integration step. At this stage the system dimensions and initial conditions \
are checked for consistency."
}], "BulletedList"],

Cell["Invoke the base numerical integration method at each step.", \
"BulletedList"],

Cell["\<\
Perform an orthogonal projection. This performs various tests such \
as checking that the basic integration proceeded correctly and that the \
Schulz iteration converges.\
\>", "BulletedList"],

Cell[TextData[{
  "Options can be used to modify the stopping criteria for the Schulz \
iteration. One option provided by our code is ",
  StyleBox["IterationSafetyFactor ", "MR"],
  "which allows control over the tolerance \[Tau] of the iteration. The \
factor is combined with a Unit in the Last Place, determined according to the \
working precision used in the integration (",
  Cell[BoxData[
      \(TraditionalForm\`ULP \[TildeTilde] 2.22045\[Cross]10\^\(-16\)\)]],
  " for IEEE double precision). "
}], "Text"],

Cell[TextData[{
  "The Frobenius norm used for the stopping criterion can be computed \
efficiently using the LAPACK LANGE functions [",
  ButtonBox["LAPACK99",
    ButtonData:>{"References.nb", "LAPACK99"},
    ButtonStyle->"Hyperlink"],
  "]"
}], "Text"],

Cell[TextData[{
  "The option ",
  StyleBox["MaxIterations", "MR"],
  " controls the maximum number of iterations ",
  Cell[BoxData[
      \(TraditionalForm\`imax\)]],
  " that should be carried out."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
  CellTags->"s:4"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {
            StyleBox["Dimensions",
              "MR"], 
            StyleBox[\({}\),
              
              "MR"], \(Specifies\ the\ dimensions\ of\ the\ matrix\ \
differential\ system\)},
          {
            StyleBox["IterationSafetyFactor",
              "MR"], 
            FractionBox[
              StyleBox["1",
                "MR"], "10"], 
            RowBox[{
            "Specifies", " ", "the", " ", "safety", " ", "factor", " ", "to", 
              " ", "use", " ", "in", " ", "the", " ", "termination", " ", 
              "criterion", " ", "for", " ", "the", " ", "Schulz", " ", 
              "iteration", " ", 
              RowBox[{
                RowBox[{"(", 
                  CounterBox["NumberedEquation", "SchulzIteration"], ")"}], 
                "."}]}]},
          {
            StyleBox["MaxIterations",
              "MR"], 
            StyleBox["Automatic",
              "MR"], 
            RowBox[{
            "Specifies", " ", "the", " ", "maximum", " ", "number", " ", "of",
               " ", "iterations", " ", "to", " ", "use", " ", "in", " ", 
              "the", " ", "Schulz", " ", "iteration", " ", 
              RowBox[{
                RowBox[{"(", 
                  CounterBox["NumberedEquation", "SchulzIteration"], ")"}], 
                "."}]}]},
          {
            StyleBox["Method",
              "MR"], 
            StyleBox["StiffnessSwitching",
              
              "MR"], \(Specifies\ the\ method\ to\ use\ for\ the\ numerical\ \
\(\(integration\)\(.\)\)\)}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  "Options of the method ",
  StyleBox["OrthogonalProjection", "MR"],
  ". "
}], "Caption"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.1 for X",
ScreenRectangle->{{0, 1600}, {0, 1200}},
AutoGeneratedPackage->None,
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{650, 500},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"OrthogonalProjection",
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
Magnification->1,
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "c:1"->{
    Cell[1776, 53, 58, 1, 95, "Chapter",
      CellTags->"c:1"]},
  "s:1"->{
    Cell[1859, 58, 50, 1, 66, "Section",
      CellTags->"s:1"]},
  "SchulzIteration"->{
    Cell[9268, 312, 184, 4, 70, "NumberedEquation",
      CellTags->"SchulzIteration"]},
  "s:2"->{
    Cell[14433, 483, 46, 1, 70, "Section",
      CellTags->"s:2"]},
  "s:3"->{
    Cell[255878, 6115, 52, 1, 70, "Section",
      CellTags->"s:3"]},
  "s:4"->{
    Cell[257897, 6177, 52, 1, 70, "Section",
      CellTags->"s:4"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"c:1", 260701, 6260},
  {"s:1", 260778, 6263},
  {"SchulzIteration", 260867, 6266},
  {"s:2", 260967, 6269},
  {"s:3", 261046, 6272},
  {"s:4", 261127, 6275}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 58, 1, 95, "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{
Cell[1859, 58, 50, 1, 66, "Section",
  CellTags->"s:1"],
Cell[1912, 61, 58, 0, 24, "Text"],
Cell[1973, 63, 104, 2, 30, "DisplayFormula"],
Cell[2080, 67, 600, 16, 43, "Text"],
Cell[2683, 85, 722, 18, 72, "Text"],
Cell[3408, 105, 209, 4, 40, "Text"],
Cell[3620, 111, 462, 9, 72, "Text"],

Cell[CellGroupData[{
Cell[4107, 124, 37, 0, 40, "Subsection"],
Cell[4147, 126, 464, 13, 56, "Text"],
Cell[4614, 141, 50, 0, 70, "BulletedList"],
Cell[4667, 143, 40, 0, 70, "BulletedList"],
Cell[4710, 145, 52, 0, 70, "BulletedList"],
Cell[4765, 147, 340, 6, 70, "Text"],
Cell[5108, 155, 173, 4, 70, "Text"],
Cell[5284, 161, 1595, 54, 70, "Text"],
Cell[6882, 217, 984, 36, 70, "Text"],
Cell[7869, 255, 88, 3, 70, "Text"],
Cell[7960, 260, 496, 13, 70, "DisplayFormula"],
Cell[8459, 275, 163, 6, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[8659, 286, 38, 0, 70, "Subsection"],
Cell[8700, 288, 393, 15, 70, "Text"],
Cell[9096, 305, 117, 3, 70, "Text"],
Cell[9216, 310, 49, 0, 70, "Text"],
Cell[9268, 312, 184, 4, 70, "NumberedEquation",
  CellTags->"SchulzIteration"],
Cell[9455, 318, 349, 9, 70, "Text"],
Cell[9807, 329, 1079, 25, 70, "Text"],

Cell[CellGroupData[{
Cell[10911, 358, 45, 0, 70, "Subsubsection"],
Cell[10959, 360, 430, 9, 70, "Text"],
Cell[11392, 371, 71, 0, 70, "Text"],
Cell[11466, 373, 780, 29, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[12283, 407, 46, 0, 70, "Subsubsection"],
Cell[12332, 409, 587, 17, 70, "Text"],
Cell[12922, 428, 221, 6, 70, "Text"],
Cell[13146, 436, 940, 32, 70, "Text"],
Cell[14089, 470, 283, 6, 70, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[14433, 483, 46, 1, 70, "Section",
  CellTags->"s:2"],

Cell[CellGroupData[{
Cell[14504, 488, 50, 0, 70, "Subsection"],
Cell[14557, 490, 298, 10, 70, "Text"],
Cell[14858, 502, 110, 2, 70, "Input"],
Cell[14971, 506, 278, 9, 70, "Text"],
Cell[15252, 517, 192, 5, 70, "Input"],
Cell[15447, 524, 128, 3, 70, "Text"],
Cell[15578, 529, 518, 10, 70, "Input"],
Cell[16099, 541, 1100, 20, 70, "Input"]
}, Open  ]],

Cell[CellGroupData[{
Cell[17236, 566, 36, 0, 70, "Subsection"],
Cell[17275, 568, 331, 12, 70, "Text"],
Cell[17609, 582, 58, 0, 70, "Text"],
Cell[17670, 584, 200, 4, 70, "DisplayFormula"],
Cell[17873, 590, 20, 0, 70, "Text"],
Cell[17896, 592, 383, 10, 70, "DisplayFormula"],
Cell[18282, 604, 19, 0, 70, "Text"],
Cell[18304, 606, 83, 1, 70, "DisplayFormula"],
Cell[18390, 609, 40, 0, 70, "Text"],
Cell[18433, 611, 216, 5, 70, "DisplayFormula"],
Cell[18652, 618, 642, 24, 70, "Text"],
Cell[19297, 644, 1558, 33, 70, "Input"],
Cell[20858, 679, 441, 8, 70, "Text"],
Cell[21302, 689, 150, 4, 70, "Input"],
Cell[21455, 695, 219, 4, 70, "Text"],

Cell[CellGroupData[{
Cell[21699, 703, 182, 4, 70, "Input"],
Cell[21884, 709, 35503, 922, 70, 9935, 601, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[57402, 1634, 289, 5, 70, "Text"],
Cell[57694, 1641, 264, 6, 70, "Input"],
Cell[57961, 1649, 175, 4, 70, "Text"],

Cell[CellGroupData[{
Cell[58161, 1657, 179, 4, 70, "Input"],
Cell[58343, 1663, 45456, 1057, 70, 10125, 616, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[103814, 2723, 162, 4, 70, "Text"],
Cell[103979, 2729, 41312, 892, 70, 6738, 460, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],

Cell[CellGroupData[{
Cell[145328, 3626, 41, 0, 70, "Subsection"],
Cell[145372, 3628, 393, 8, 70, "Text"],
Cell[145768, 3638, 588, 19, 70, "Text"],
Cell[146359, 3659, 238, 5, 70, "Text"],
Cell[146600, 3666, 168, 6, 70, "Text"],
Cell[146771, 3674, 233, 5, 70, "DisplayFormula"],
Cell[147007, 3681, 488, 16, 70, "Text"],
Cell[147498, 3699, 47, 0, 70, "Text"],
Cell[147548, 3701, 495, 13, 70, "DisplayFormula"],
Cell[148046, 3716, 36, 0, 70, "Text"],
Cell[148085, 3718, 340, 9, 70, "DisplayFormula"],
Cell[148428, 3729, 205, 6, 70, "Text"],
Cell[148636, 3737, 192, 4, 70, "DisplayFormula"],
Cell[148831, 3743, 137, 3, 70, "Text"],
Cell[148971, 3748, 1326, 26, 70, "Input"],
Cell[150300, 3776, 118, 3, 70, "Text"],
Cell[150421, 3781, 147, 3, 70, "Input"],
Cell[150571, 3786, 90, 3, 70, "Text"],
Cell[150664, 3791, 244, 6, 70, "Input"],
Cell[150911, 3799, 119, 3, 70, "Text"],

Cell[CellGroupData[{
Cell[151055, 3806, 107, 2, 70, "Input"],
Cell[151165, 3810, 121, 3, 70, "Output"]
}, Open  ]],
Cell[151301, 3816, 116, 3, 70, "Text"],

Cell[CellGroupData[{
Cell[151442, 3823, 87, 2, 70, "Input"],
Cell[151532, 3827, 41516, 980, 70, 9720, 582, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]],
Cell[193063, 4810, 171, 4, 70, "Text"],
Cell[193237, 4816, 367, 8, 70, "Input"],
Cell[193607, 4826, 229, 5, 70, "Text"],

Cell[CellGroupData[{
Cell[193861, 4835, 108, 2, 70, "Input"],
Cell[193972, 4839, 120, 3, 70, "Output"]
}, Open  ]],
Cell[194107, 4845, 150, 3, 70, "Text"],

Cell[CellGroupData[{
Cell[194282, 4852, 86, 2, 70, "Input"],
Cell[194371, 4856, 61446, 1252, 70, 10013, 612, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[255878, 6115, 52, 1, 70, "Section",
  CellTags->"s:3"],
Cell[255933, 6118, 141, 4, 70, "Text"],
Cell[256077, 6124, 497, 9, 70, "BulletedList"],
Cell[256577, 6135, 84, 1, 70, "BulletedList"],
Cell[256664, 6138, 202, 4, 70, "BulletedList"],
Cell[256869, 6144, 517, 10, 70, "Text"],
Cell[257389, 6156, 256, 7, 70, "Text"],
Cell[257648, 6165, 212, 7, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[257897, 6177, 52, 1, 70, "Section",
  CellTags->"s:4"],
Cell[257952, 6180, 1812, 47, 70, "DefinitionBox3Col"],
Cell[259767, 6229, 107, 4, 70, "Caption"]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

