(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.1'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[   7325835,     318886]*)
(*NotebookOutlinePosition[   7377113,     320068]*)
(*  CellTagsIndexPosition[   7375503,     320016]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Partial Differential Equations", "Chapter"],

Cell[CellGroupData[{

Cell["The Numerical Method Of Lines", "Section",
  CellTags->{"c:1", "PDE:1"}],

Cell[CellGroupData[{

Cell["Introduction", "Subsection",
  CellTags->{"c:2", "b:7.0", "ndsg:2.0.0", "PDE:1.1"}],

Cell["\<\
The numerical methods of lines is a technique for solving partial \
differential equations discretizing in all but one dimension, and then \
integrating the semi-discrete problem as a system of ODEs or DAEs. A \
significant advantage of the method is that it allows the solution to take \
advantage of the sophisticated general purpose methods and software that have \
been developed for numerically integrating ODEs and DAEs. For the PDEs to \
which the method of lines is applicable, the method typically proves to be \
quite efficient. \
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["\<\
It is necessary that the PDE problem be well-posed as an initial \
value (Cauchy) problem in at least one dimension, since the ODE and DAE \
integrators used are initial value problem solvers. This rules out purely \
elliptic equations such as Laplace's equation, but leaves a large class of \
evolution equations that can be solved quite efficiently.\
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["\<\
A simple example illustrates better than mere words the fundamental \
idea of the method. Consider the following problem (a simple model for \
seasonal variation of heat in soil). \
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    \(TraditionalForm\`u\_t == \(1\/8\) u\_xx, 
    u(0, t) == sin(2  \[Pi]\ t), \(u\_x\)(1, t) == 0, \ 
    u(x, 0)\  \[Equal] \ 0\)], "NumberedEquation",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["\<\
This is a candidate for the method of lines since we have the \
initial value u(x,0) == 0.\
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["\<\
Problem (1) will be discretized with respect to the variable x \
using second order finite differences, in particular using the \
approximation\
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    \(TraditionalForm\`\(u\_xx\)(x, 
        t)\  \[TildeEqual] \(u(x\  + \ h, \ t)\  - \ 2\ \(u(x, \ t)\)\  - \ \
u(x\  - \ h, \ t)\)\/h\^2\)], "NumberedEquation",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["\<\
Even though finite difference discretizations are the most common, \
there is certainly no requirement that discretizations for the method of \
lines be done with finite differences; finite volume or even finite element \
discretizations can also be used.\
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "To use the discretization above, choose a uniform grid ",
  Cell[BoxData[
      \(TraditionalForm\`x\_i, \ 0\  \[LessEqual] \ i\  \[LessEqual] \ n\)]],
  "with spacing h == 1/n such that ",
  Cell[BoxData[
      \(TraditionalForm\`x\_i\  \[Equal] \ i\ h\)]],
  ". Let ",
  Cell[BoxData[
      \(TraditionalForm\`u\_i[t]\)]],
  " be the value of ",
  Cell[BoxData[
      \(TraditionalForm\`u(x\_i, t)\)]],
  ".",
  " ",
  "For the purposes of illustrating the problem set-up, a particular value of \
n is chosen."
}], "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["\<\
This defines a particular value of n and the corresponding value of \
h used in the subsequent commands. This can be changed to make a finer or \
coarser spatial approximation.\
\>", "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    \(n\  = \ 10; h\_n\  = \ 1\/n;\)], "Input",
  CellLabel->"In[3]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "This defines the vector of ",
  Cell[BoxData[
      \(TraditionalForm\`u\_i\)]],
  "."
}], "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(U[t_]\  = \ Table[u\_i[t], {i, 0, n}]\)], "Input",
  CellLabel->"In[4]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    \({u\_0[t], u\_1[t], u\_2[t], u\_3[t], u\_4[t], u\_5[t], u\_6[t], 
      u\_7[t], u\_8[t], u\_9[t], u\_10[t]}\)], "Output",
  CellLabel->"Out[4]="]
}, Open  ]],

Cell["\<\
For 1 \[LessEqual] i \[LessEqual] 9, we can use the centered \
difference formula (2) to obtain a system of ODEs. However, before doing \
this, it is useful to incorporate the boundary conditions first.\
\>", "Text",\

  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "The Dirichlet boundary condition at x == 0 can easily be handled by simply \
defining ",
  Cell[BoxData[
      \(TraditionalForm\`u\_0\)]],
  "as a function of t. An alternative option is to differentiate the boundary \
condition with respect to time and use the corresponding differential \
equation.",
  " ",
  "In this example, the latter method will be used."
}], "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "The Neumann boundary condition at x == 1 is a little more difficult. With \
second order differences, one way to handle it is with reflection: imagine we \
are solving the problem on the interval 0 \[LessEqual] x \[LessEqual] 2 with \
the same boundary conditions at x == 0 and x == 2.",
  " ",
  "Since the initial condition and boundary conditions are symmetric with \
respect to x, the solution should be symmetric with respect to x for all \
time, and so symmetry is equivalent to the Neumann boundary condition at x == \
1. Thus,",
  StyleBox[" ", "DisplayMath"],
  Cell[BoxData[
      \(TraditionalForm\`u(1\  + \ h, \ t)\  \[Equal] \ 
        u(1\  - \ h, \ t)\)]],
  ", so ",
  Cell[BoxData[
      \(TraditionalForm\`u\_\(n + 1\)[t]\  \[Equal] \ u\_\(n - 1\)[t]\)]]
}], "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "This uses ListCorrelate to apply the difference formula. The padding ",
  Cell[BoxData[
      \({u\_\(n - 1\)[t]}\)]],
  "implements the Neumann boundary condition."
}], "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqns\  = \ 
      Thread[D[U[t], t]\  \[Equal] \ 
          Join[{D[Sin[2  \[Pi]\ t], t]}, 
            ListCorrelate[{1, \(-2\), 1}/h\_n\^2, \ 
                U[t], {1, 2}, {u\_\(n - 1\)[t]}]/8]]\)], "Input",
  CellLabel->"In[5]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_0\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(2\ \[Pi]\ Cos[2\ \[Pi]\ t]\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_1\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_0[t] - 200\ u\_1[t] + 
                100\ u\_2[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_2\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_1[t] - 200\ u\_2[t] + 
                100\ u\_3[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_3\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_2[t] - 200\ u\_3[t] + 
                100\ u\_4[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_4\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_3[t] - 200\ u\_4[t] + 
                100\ u\_5[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_5\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_4[t] - 200\ u\_5[t] + 
                100\ u\_6[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_6\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_5[t] - 200\ u\_6[t] + 
                100\ u\_7[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_7\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_6[t] - 200\ u\_7[t] + 
                100\ u\_8[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_8\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_7[t] - 200\ u\_8[t] + 
                100\ u\_9[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_9\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((100\ u\_8[t] - 200\ u\_9[t] + 
                100\ u\_10[t])\)\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(u\_10\), "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], 
          "\[Equal]", \(1\/8\ \((200\ u\_9[t] - 200\ u\_10[t])\)\)}]}], 
      "}"}]], "Output",
  CellLabel->"Out[5]="]
}, Open  ]],

Cell["This sets up the zero initial condition.", "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(initc\  = \ Thread[U[0]\  \[Equal] \ Table[0, {n + 1}]]\)], "Input",
  CellLabel->"In[6]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    \({u\_0[0] \[Equal] 0, u\_1[0] \[Equal] 0, u\_2[0] \[Equal] 0, 
      u\_3[0] \[Equal] 0, u\_4[0] \[Equal] 0, u\_5[0] \[Equal] 0, 
      u\_6[0] \[Equal] 0, u\_7[0] \[Equal] 0, u\_8[0] \[Equal] 0, 
      u\_9[0] \[Equal] 0, u\_10[0] \[Equal] 0}\)], "Output",
  CellLabel->"Out[6]="]
}, Open  ]],

Cell["\<\
Now the PDE has been partially discretized into an ODE initial \
value problem that can be solved by the ODE integrators in NDSolve.\
\>", \
"Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["This solves the ODE initial value problem.", "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(lines\  = \ NDSolve[{eqns, \ initc}, U[t], {t, 0, 4}]\)], "Input",
  CellLabel->"In[7]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(u\_0[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_1[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_2[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_3[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_4[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_5[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_6[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_7[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_8[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_9[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}], ",", 
          RowBox[{\(u\_10[t]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 4.`}}, "<>"]\),
                False,
                Editable->False], "[", "t", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[7]="]
}, Open  ]],

Cell[TextData[{
  "This shows the solutions ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(u(x\_i, t)\)\(\ \)\)\)]],
  "plotted as a function of x and t."
}], "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Show[\[IndentingNewLine]Block[{$DisplayFunction\  = \ 
              Identity}, \[IndentingNewLine]Table[
            ParametricPlot3D[{i\ h\_n, t, \ First[u\_i[t]\  /. lines]}, {t, 
                0, 4}], {i, 0, n}]], \ PlotRange \[Rule] All, \ 
        AxesLabel \[Rule] {"\<x\>", "\<t\>", "\<u\>"}];\)\)], "Input",
  CellLabel->"In[8]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1.3598 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics3D
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.18377 1.46228 4.87891e-017 1.46228 [
[.04503 .1059 -5.69635 -9 ]
[.04503 .1059 .30365 0 ]
[.11707 .07724 -22.4941 -9 ]
[.11707 .07724 1.50586 0 ]
[.19016 .04816 -16.6522 -9 ]
[.19016 .04816 1.34783 0 ]
[.26431 .01866 -21.9116 -9 ]
[.26431 .01866 2.08837 0 ]
[.33955 -0.01126 -5.40509 -9 ]
[.33955 -0.01126 .59491 0 ]
[.13096 -0.02147 -9.2512 -12.5625 ]
[.13096 -0.02147 .7488 0 ]
[.00356 .72009 -6 -2.97599 ]
[.00356 .72009 0 6.02401 ]
[.19968 .90362 -6 -3.0863 ]
[.19968 .90362 0 5.9137 ]
[.37564 1.06829 -6 -3.18171 ]
[.37564 1.06829 0 5.81829 ]
[.5344 1.21685 -6 -3.26507 ]
[.5344 1.21685 0 5.73493 ]
[.67836 1.35156 -6 -3.3385 ]
[.67836 1.35156 0 5.6615 ]
[.28794 1.09398 -10 -4.44114 ]
[.28794 1.09398 0 8.12136 ]
[.03205 .14123 -12 -2.72059 ]
[.03205 .14123 0 6.27941 ]
[.02084 .26827 -24 -2.77567 ]
[.02084 .26827 0 6.22433 ]
[.00912 .40112 -6 -2.83339 ]
[.00912 .40112 0 6.16661 ]
[-0.00315 .54018 -18 -2.89396 ]
[-0.00315 .54018 0 6.10604 ]
[-0.016 .6859 -6 -2.95759 ]
[-0.016 .6859 0 6.04241 ]
[-0.07658 .43286 -10 -3.95494 ]
[-0.07658 .43286 0 8.60756 ]
[ 0 0 0 0 ]
[ 1 1.3598 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.05012 .12232 m
.35855 0 L
s
.05725 .11949 m
.06336 .12629 L
s
[(0)] .04503 .1059 .89878 1 Mshowa
.12911 .091 m
.13512 .09788 L
s
[(0.25)] .11707 .07724 .87451 1 Mshowa
.202 .06209 m
.20792 .06905 L
s
[(0.5)] .19016 .04816 .85024 1 Mshowa
.27595 .03276 m
.28177 .0398 L
s
[(0.75)] .26431 .01866 .82597 1 Mshowa
.35099 .003 m
.3567 .01013 L
s
[(1)] .33955 -0.01126 .8017 1 Mshowa
.125 Mabswid
.07154 .11383 m
.07519 .11792 L
s
.08587 .10814 m
.08951 .11224 L
s
.10024 .10244 m
.10387 .10655 L
s
.11465 .09673 m
.11827 .10085 L
s
.1436 .08525 m
.1472 .08938 L
s
.15814 .07948 m
.16173 .08363 L
s
.17272 .0737 m
.17629 .07786 L
s
.18734 .0679 m
.1909 .07207 L
s
.21671 .05625 m
.22025 .06044 L
s
.23145 .05041 m
.23498 .0546 L
s
.24624 .04454 m
.24976 .04875 L
s
.26108 .03866 m
.26458 .04287 L
s
.29087 .02684 m
.29435 .03108 L
s
.30584 .02091 m
.3093 .02515 L
s
.32084 .01495 m
.3243 .01921 L
s
.33589 .00899 m
.33933 .01325 L
s
gsave
.13096 -0.02147 -70.2512 -16.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
69.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
0 .69467 m
.70973 1.3598 L
s
.02087 .71423 m
.02953 .7113 L
s
[(0)] .00356 .72009 1 -0.33867 Mshowa
.21712 .89815 m
.22584 .89541 L
s
[(1)] .19968 .90362 1 -0.31416 Mshowa
.39318 1.06315 m
.40195 1.06058 L
s
[(2)] .37564 1.06829 1 -0.29295 Mshowa
.55203 1.21201 m
.56084 1.20959 L
s
[(3)] .5344 1.21685 1 -0.27443 Mshowa
.69606 1.34699 m
.70491 1.34471 L
s
[(4)] .67836 1.35156 1 -0.25811 Mshowa
.125 Mabswid
.0619 .75268 m
.0671 .75095 L
s
.10201 .79027 m
.10722 .78856 L
s
.14123 .82703 m
.14645 .82534 L
s
.17959 .86298 m
.18481 .86131 L
s
.25384 .93256 m
.25908 .93094 L
s
.28979 .96625 m
.29503 .96465 L
s
.32498 .99923 m
.33023 .99764 L
s
.35943 1.03152 m
.36469 1.02996 L
s
.42625 1.09413 m
.43151 1.09261 L
s
.45864 1.12449 m
.46392 1.12299 L
s
.49039 1.15425 m
.49567 1.15276 L
s
.52151 1.18341 m
.5268 1.18195 L
s
.58195 1.24005 m
.58724 1.23862 L
s
.6113 1.26756 m
.6166 1.26614 L
s
.64009 1.29454 m
.64539 1.29314 L
s
.66834 1.32101 m
.67364 1.31963 L
s
gsave
.28794 1.09398 -71 -8.44114 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(t) show
69.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.05012 .12232 m
0 .69467 L
s
.04905 .1345 m
.05755 .13114 L
s
[(-1)] .03205 .14123 1 -0.39542 Mshowa
.03791 .26173 m
.04644 .25846 L
s
[(-0.5)] .02084 .26827 1 -0.38318 Mshowa
.02626 .39477 m
.03483 .39159 L
s
[(0)] .00912 .40112 1 -0.37036 Mshowa
.01407 .53404 m
.02267 .53096 L
s
[(0.5)] -0.00315 .54018 1 -0.3569 Mshowa
.00129 .67998 m
.00993 .67701 L
s
[(1)] -0.016 .6859 1 -0.34276 Mshowa
.125 Mabswid
.04686 .1595 m
.05196 .1575 L
s
.04465 .18472 m
.04976 .18273 L
s
.04243 .21016 m
.04754 .20818 L
s
.04018 .23583 m
.04529 .23386 L
s
.03562 .28786 m
.04075 .28591 L
s
.03331 .31422 m
.03844 .31228 L
s
.03098 .34083 m
.03612 .3389 L
s
.02863 .36768 m
.03377 .36576 L
s
.02387 .42211 m
.02901 .42022 L
s
.02145 .4497 m
.0266 .44782 L
s
.01901 .47755 m
.02417 .47569 L
s
.01655 .50566 m
.02171 .50381 L
s
.01156 .56267 m
.01673 .56084 L
s
.00903 .59158 m
.0142 .58977 L
s
.00647 .62077 m
.01165 .61896 L
s
.00389 .65023 m
.00907 .64844 L
s
gsave
-0.07658 .43286 -71 -7.95494 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(u) show
69.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.05012 .12232 m
0 .69467 L
s
0 .69467 m
.70973 1.3598 L
s
.70973 1.3598 m
.69754 .84072 L
s
.69754 .84072 m
.05012 .12232 L
s
.35855 0 m
.96518 .75898 L
s
.96518 .75898 m
1 1.28532 L
s
1 1.28532 m
.33887 .579 L
s
.33887 .579 m
.35855 0 L
s
.05012 .12232 m
0 .69467 L
s
0 .69467 m
.33887 .579 L
s
.33887 .579 m
.35855 0 L
s
.35855 0 m
.05012 .12232 L
s
.69754 .84072 m
.96518 .75898 L
s
.96518 .75898 m
1 1.28532 L
s
1 1.28532 m
.70973 1.3598 L
s
.70973 1.3598 m
.69754 .84072 L
s
0 0 m
1 0 L
1 1.3598 L
0 1.3598 L
closepath
clip
newpath
.5 Mabswid
.66362 .81034 m
.6712 .84739 L
s
.65652 .79921 m
.66362 .81034 L
s
.64982 .81441 m
.65652 .79921 L
s
.6712 .84739 m
.67928 .90759 L
s
.6434 .85392 m
.64982 .81441 L
s
.67928 .90759 m
.68782 .98581 L
s
.70524 .91875 m
.71292 .95257 L
s
.69795 .90003 m
.70524 .91875 L
s
.69101 .89758 m
.69795 .90003 L
s
.71292 .95257 m
.72094 .99877 L
s
.68782 .98581 m
.69672 1.07479 L
s
.68435 .91093 m
.69101 .89758 L
s
.63711 .91333 m
.6434 .85392 L
s
.73937 .96838 m
.74672 .98793 L
s
.7323 .95844 m
.73937 .96838 L
s
.6779 .93802 m
.68435 .91093 L
s
.72546 .95842 m
.7323 .95844 L
s
.76652 .9953 m
.77344 1.00155 L
s
.71882 .96754 m
.72546 .95842 L
s
.75975 .99491 m
.76652 .9953 L
s
.63081 .986 m
.63711 .91333 L
s
.67152 .97527 m
.6779 .93802 L
s
.7123 .98405 m
.71882 .96754 L
s
.79384 1.01619 m
.80056 1.01794 L
s
.7531 .99962 m
.75975 .99491 L
s
.78718 1.01718 m
.79384 1.01619 L
s
.7465 1.00809 m
.7531 .99962 L
s
.82109 1.02646 m
.82776 1.0283 L
s
.7058 1.00533 m
.7123 .98405 L
s
.78052 1.01996 m
.78718 1.01718 L
s
.66511 1.01789 m
.67152 .97527 L
s
.8144 1.02538 m
.82109 1.02646 L
s
.84819 1.02757 m
.85488 1.0312 L
s
.73989 1.01854 m
.7465 1.00809 L
s
.77384 1.02337 m
.78052 1.01996 L
s
.80767 1.02432 m
.8144 1.02538 L
s
.84146 1.02375 m
.84819 1.02757 L
s
.87521 1.02284 m
.88193 1.02837 L
s
.69924 1.02819 m
.7058 1.00533 L
s
.52387 .6505 m
.53151 .67586 L
s
.51611 .65211 m
.52387 .6505 L
s
.62435 1.06335 m
.63081 .986 L
s
.53151 .67586 m
.53911 .72659 L
s
.73321 1.02893 m
.73989 1.01854 L
s
.90224 1.01527 m
.90898 1.0222 L
s
.86843 1.01682 m
.87521 1.02284 L
s
.76707 1.02614 m
.77384 1.02337 L
s
.83465 1.01931 m
.84146 1.02375 L
s
.80087 1.02256 m
.80767 1.02432 L
s
.50811 .67988 m
.51611 .65211 L
s
.53911 .72659 m
.54678 .79868 L
s
.92936 1.00691 m
.93609 1.01465 L
s
.89545 1.00787 m
.90224 1.01527 L
s
.86159 1.01013 m
.86843 1.01682 L
s
.65855 1.06023 m
.66511 1.01789 L
s
.82776 1.01388 m
.83465 1.01931 L
s
.79397 1.0194 m
.80087 1.02256 L
s
.76019 1.02709 m
.76707 1.02614 L
s
.69254 1.04912 m
.69924 1.02819 L
s
.7264 1.03718 m
.73321 1.02893 L
s
.95663 .99884 m
.96332 1.00685 L
s
.92258 .9988 m
.92936 1.00691 L
s
.88859 .99997 m
.89545 1.00787 L
s
.85467 1.00267 m
.86159 1.01013 L
s
.82079 1.00722 m
.82776 1.01388 L
s
.78696 1.01433 m
.79397 1.0194 L
s
.91575 .99036 m
.92258 .9988 L
s
.94989 .99049 m
.95663 .99884 L
s
.88168 .9916 m
.88859 .99997 L
s
.84768 .9944 m
.85467 1.00267 L
s
.75317 1.0252 m
.76019 1.02709 L
s
.81372 .9992 m
.82079 1.00722 L
s
.71941 1.04142 m
.7264 1.03718 L
s
.77983 1.00704 m
.78696 1.01433 L
s
.90887 .98167 m
.91575 .99036 L
s
.9431 .98188 m
.94989 .99049 L
s
.54678 .79868 m
.55458 .88582 L
s
.87472 .98283 m
.88168 .9916 L
s
.49982 .73056 m
.50811 .67988 L
s
.84062 .98539 m
.84768 .9944 L
s
.80657 .98985 m
.81372 .9992 L
s
.68563 1.06475 m
.69254 1.04912 L
s
.746 1.01978 m
.75317 1.0252 L
s
.56548 .75955 m
.57323 .78721 L
s
.55776 .74813 m
.56548 .75955 L
s
.57323 .78721 m
.58107 .82912 L
s
.90196 .97285 m
.90887 .98167 L
s
.93627 .9731 m
.9431 .98188 L
s
.77258 .99747 m
.77983 1.00704 L
s
.86772 .97379 m
.87472 .98283 L
s
.83351 .97579 m
.84062 .98539 L
s
.65174 1.09642 m
.65855 1.06023 L
s
.79934 .97936 m
.80657 .98985 L
s
.55001 .75331 m
.55776 .74813 L
s
.6176 1.13557 m
.62435 1.06335 L
s
.71223 1.04023 m
.71941 1.04142 L
s
.73867 1.01057 m
.746 1.01978 L
s
.89502 .96402 m
.90196 .97285 L
s
.76524 .98584 m
.77258 .99747 L
s
.92941 .96428 m
.93627 .9731 L
s
.86068 .96461 m
.86772 .97379 L
s
.58107 .82912 m
.58902 .88176 L
s
.82635 .96581 m
.83351 .97579 L
s
.79206 .96803 m
.79934 .97936 L
s
.75781 .97257 m
.76524 .98584 L
s
.73121 .99771 m
.73867 1.01057 L
s
.88805 .95528 m
.89502 .96402 L
s
.92252 .95553 m
.92941 .96428 L
s
.8536 .95546 m
.86068 .96461 L
s
.81916 .95568 m
.82635 .96581 L
s
.70484 1.0328 m
.71223 1.04023 L
s
.67848 1.07228 m
.68563 1.06475 L
s
.78472 .95627 m
.79206 .96803 L
s
.54218 .77379 m
.55001 .75331 L
s
.75033 .9583 m
.75781 .97257 L
s
.72364 .9818 m
.73121 .99771 L
s
.60785 .83154 m
.61566 .85657 L
s
.60012 .81582 m
.60785 .83154 L
s
.61566 .85657 m
.62356 .88906 L
s
.58902 .88176 m
.59709 .94035 L
s
.88106 .94675 m
.88805 .95528 L
s
.84651 .94648 m
.8536 .95546 L
s
.91561 .94694 m
.92252 .95553 L
s
.81194 .94567 m
.81916 .95568 L
s
.77737 .94453 m
.78472 .95627 L
s
.716 .96377 m
.72364 .9818 L
s
.74282 .94376 m
.75033 .9583 L
s
.55458 .88582 m
.56255 .97965 L
s
.69728 1.01909 m
.70484 1.0328 L
s
.59245 .81022 m
.60012 .81582 L
s
.62356 .88906 m
.63153 .92636 L
s
.70832 .94478 m
.716 .96377 L
s
.7353 .92971 m
.74282 .94376 L
s
.77 .93323 m
.77737 .94453 L
s
.70064 .9261 m
.70832 .94478 L
s
.68957 .99978 m
.69728 1.01909 L
s
.80471 .93603 m
.81194 .94567 L
s
.8394 .9378 m
.84651 .94648 L
s
.87405 .93851 m
.88106 .94675 L
s
.90868 .93861 m
.91561 .94694 L
s
.65822 .89931 m
.66605 .92389 L
s
.65045 .87833 m
.65822 .89931 L
s
.66605 .92389 m
.67391 .95023 L
s
.64463 1.12107 m
.65174 1.09642 L
s
.67106 1.0698 m
.67848 1.07228 L
s
.67391 .95023 m
.68176 .97622 L
s
.49117 .79852 m
.49982 .73056 L
s
.69298 .90896 m
.70064 .9261 L
s
.63153 .92636 m
.63956 .96518 L
s
.68176 .97622 m
.68957 .99978 L
s
.64275 .8624 m
.65045 .87833 L
s
.72779 .91683 m
.7353 .92971 L
s
.58478 .81449 m
.59245 .81022 L
s
.76263 .92278 m
.77 .93323 L
s
.68536 .89438 m
.69298 .90896 L
s
.63511 .85239 m
.64275 .8624 L
s
.79746 .92695 m
.80471 .93603 L
s
.83227 .92951 m
.8394 .9378 L
s
.86703 .93057 m
.87405 .93851 L
s
.90174 .93058 m
.90868 .93861 L
s
.59709 .94035 m
.60523 .99923 L
s
.63956 .96518 m
.6476 1.00199 L
s
.72031 .9057 m
.72779 .91683 L
s
.67778 .88311 m
.68536 .89438 L
s
.53421 .80669 m
.54218 .77379 L
s
.66341 1.05663 m
.67106 1.0698 L
s
.75527 .91344 m
.76263 .92278 L
s
.62751 .84852 m
.63511 .85239 L
s
.79021 .91856 m
.79746 .92695 L
s
.6476 1.00199 m
.65557 1.03341 L
s
.82512 .92164 m
.83227 .92951 L
s
.71285 .89666 m
.72031 .9057 L
s
.85998 .92296 m
.86703 .93057 L
s
.89478 .92286 m
.90174 .93058 L
s
.67024 .87551 m
.67778 .88311 L
s
.65557 1.03341 m
.66341 1.05663 L
s
.57709 .82735 m
.58478 .81449 L
s
.74791 .90538 m
.75527 .91344 L
s
.6105 1.19275 m
.6176 1.13557 L
s
.78295 .9109 m
.79021 .91856 L
s
.7054 .88983 m
.71285 .89666 L
s
.81796 .91418 m
.82512 .92164 L
s
.61993 .85033 m
.62751 .84852 L
s
.8878 .91542 m
.89478 .92286 L
s
.85292 .91561 m
.85998 .92296 L
s
.6372 1.13006 m
.64463 1.12107 L
s
.66273 .87153 m
.67024 .87551 L
s
.60523 .99923 m
.6134 1.05245 L
s
.74055 .89859 m
.74791 .90538 L
s
.56255 .97965 m
.57067 1.07043 L
s
.77568 .90393 m
.78295 .9109 L
s
.81077 .90707 m
.81796 .91418 L
s
.69797 .88507 m
.7054 .88983 L
s
.8808 .90819 m
.8878 .91542 L
s
.84582 .90846 m
.85292 .91561 L
s
.65522 .87072 m
.66273 .87153 L
s
.73319 .89294 m
.74055 .89859 L
s
.61234 .85675 m
.61993 .85033 L
s
.56932 .84653 m
.57709 .82735 L
s
.6134 1.05245 m
.62151 1.09452 L
s
.76839 .89753 m
.77568 .90393 L
s
.52607 .8477 m
.53421 .80669 L
s
.62948 1.12118 m
.6372 1.13006 L
s
.80356 .90019 m
.81077 .90707 L
s
.87377 .90107 m
.8808 .90819 L
s
.8387 .9014 m
.84582 .90846 L
s
.69052 .88201 m
.69797 .88507 L
s
.72581 .88814 m
.73319 .89294 L
s
.64769 .87226 m
.65522 .87072 L
s
.62151 1.09452 m
.62948 1.12118 L
s
.4822 .87595 m
.49117 .79852 L
s
.76107 .89151 m
.76839 .89753 L
s
.79631 .89341 m
.80356 .90019 L
s
.8667 .89395 m
.87377 .90107 L
s
.83153 .8943 m
.8387 .9014 L
s
.60469 .86615 m
.61234 .85675 L
s
.68305 .88007 m
.69052 .88201 L
s
.71839 .88382 m
.72581 .88814 L
s
.75371 .88561 m
.76107 .89151 L
s
.78902 .88656 m
.79631 .89341 L
s
.60301 1.22631 m
.6105 1.19275 L
s
.85959 .88671 m
.8667 .89395 L
s
.82432 .88705 m
.83153 .8943 L
s
.57067 1.07043 m
.57889 1.14808 L
s
.56145 .86902 m
.56932 .84653 L
s
.64012 .87505 m
.64769 .87226 L
s
.67554 .87854 m
.68305 .88007 L
s
.71093 .87952 m
.71839 .88382 L
s
.74631 .87957 m
.75371 .88561 L
s
.78168 .8795 m
.78902 .88656 L
s
.85243 .87925 m
.85959 .88671 L
s
.81706 .87954 m
.82432 .88705 L
s
.59697 .87655 m
.60469 .86615 L
s
.37878 .49497 m
.38638 .5348 L
s
.37042 .48214 m
.37878 .49497 L
s
.51775 .89141 m
.52607 .8477 L
s
.63248 .87783 m
.64012 .87505 L
s
.66796 .87665 m
.67554 .87854 L
s
.84521 .87146 m
.85243 .87925 L
s
.70341 .8748 m
.71093 .87952 L
s
.77429 .87208 m
.78168 .8795 L
s
.73885 .87315 m
.74631 .87957 L
s
.80975 .87167 m
.81706 .87954 L
s
.36119 .49672 m
.37042 .48214 L
s
.38638 .5348 m
.39343 .59881 L
s
.55347 .89135 m
.56145 .86902 L
s
.83795 .86329 m
.84521 .87146 L
s
.57889 1.14808 m
.5871 1.20357 L
s
.80239 .86341 m
.80975 .87167 L
s
.76685 .86421 m
.77429 .87208 L
s
.73133 .86612 m
.73885 .87315 L
s
.59518 1.23051 m
.60301 1.22631 L
s
.69583 .86925 m
.70341 .8748 L
s
.58915 .88581 m
.59697 .87655 L
s
.66031 .87367 m
.66796 .87665 L
s
.62476 .87934 m
.63248 .87783 L
s
.83062 .85473 m
.83795 .86329 L
s
.79497 .85475 m
.80239 .86341 L
s
.75935 .85582 m
.76685 .86421 L
s
.72375 .85833 m
.73133 .86612 L
s
.68817 .86254 m
.69583 .86925 L
s
.65258 .86896 m
.66031 .87367 L
s
.35102 .53664 m
.36119 .49672 L
s
.5871 1.20357 m
.59518 1.23051 L
s
.47295 .95334 m
.4822 .87595 L
s
.39343 .59881 m
.40014 .68177 L
s
.61695 .87842 m
.62476 .87934 L
s
.7875 .84571 m
.79497 .85475 L
s
.82325 .8458 m
.83062 .85473 L
s
.75179 .84693 m
.75935 .85582 L
s
.71611 .84972 m
.72375 .85833 L
s
.58123 .89188 m
.58915 .88581 L
s
.68044 .85447 m
.68817 .86254 L
s
.54537 .90995 m
.55347 .89135 L
s
.50928 .93185 m
.51775 .89141 L
s
.64477 .8621 m
.65258 .86896 L
s
.77998 .83637 m
.7875 .84571 L
s
.81583 .83654 m
.82325 .8458 L
s
.74419 .83759 m
.75179 .84693 L
s
.70841 .84031 m
.71611 .84972 L
s
.60905 .87419 m
.61695 .87842 L
s
.67265 .84501 m
.68044 .85447 L
s
.77243 .82683 m
.77998 .83637 L
s
.63687 .8529 m
.64477 .8621 L
s
.80836 .82706 m
.81583 .83654 L
s
.73653 .82789 m
.74419 .83759 L
s
.42024 .60999 m
.42814 .64652 L
s
.41202 .58924 m
.42024 .60999 L
s
.70066 .83019 m
.70841 .84031 L
s
.57322 .89304 m
.58123 .89188 L
s
.40339 .58548 m
.41202 .58924 L
s
.66479 .83425 m
.67265 .84501 L
s
.60106 .86615 m
.60905 .87419 L
s
.42814 .64652 m
.43584 .69602 L
s
.76483 .81721 m
.77243 .82683 L
s
.80085 .81747 m
.80836 .82706 L
s
.62892 .84144 m
.63687 .8529 L
s
.72884 .81797 m
.73653 .82789 L
s
.69286 .81957 m
.70066 .83019 L
s
.53717 .92158 m
.54537 .90995 L
s
.40014 .68177 m
.40675 .77616 L
s
.65688 .82247 m
.66479 .83425 L
s
.39429 .59823 m
.40339 .58548 L
s
.593 .85423 m
.60106 .86615 L
s
.6209 .82805 m
.62892 .84144 L
s
.56512 .88812 m
.57322 .89304 L
s
.33996 .59742 m
.35102 .53664 L
s
.7572 .80763 m
.76483 .81721 L
s
.79331 .80788 m
.80085 .81747 L
s
.7211 .808 m
.72884 .81797 L
s
.68501 .80866 m
.69286 .81957 L
s
.64892 .81002 m
.65688 .82247 L
s
.43584 .69602 m
.44347 .75423 L
s
.61284 .81329 m
.6209 .82805 L
s
.58489 .83883 m
.593 .85423 L
s
.5007 .96313 m
.50928 .93185 L
s
.74953 .7982 m
.7572 .80763 L
s
.78574 .79842 m
.79331 .80788 L
s
.71333 .79812 m
.7211 .808 L
s
.67713 .79775 m
.68501 .80866 L
s
.64093 .79735 m
.64892 .81002 L
s
.55696 .87668 m
.56512 .88812 L
s
.46354 .68414 m
.47167 .71446 L
s
.4553 .66252 m
.46354 .68414 L
s
.57674 .82073 m
.58489 .83883 L
s
.60475 .79786 m
.61284 .81329 L
s
.44689 .65097 m
.4553 .66252 L
s
.47167 .71446 m
.47973 .75117 L
s
.52889 .92378 m
.53717 .92158 L
s
.56857 .80103 m
.57674 .82073 L
s
.63292 .78492 m
.64093 .79735 L
s
.66922 .78708 m
.67713 .79775 L
s
.59663 .78254 m
.60475 .79786 L
s
.70554 .78848 m
.71333 .79812 L
s
.74184 .78902 m
.74953 .7982 L
s
.77813 .78917 m
.78574 .79842 L
s
.46357 1.02041 m
.47295 .95334 L
s
.54876 .85906 m
.55696 .87668 L
s
.56038 .781 m
.56857 .80103 L
s
.47973 .75117 m
.4878 .79117 L
s
.38469 .62534 m
.39429 .59823 L
s
.43825 .64978 m
.44689 .65097 L
s
.44347 .75423 m
.45113 .81574 L
s
.55218 .76193 m
.56038 .781 L
s
.51593 .75552 m
.52413 .78244 L
s
.50771 .7314 m
.51593 .75552 L
s
.54055 .83631 m
.54876 .85906 L
s
.52413 .78244 m
.53234 .81009 L
s
.5885 .76809 m
.59663 .78254 L
s
.49946 .71176 m
.50771 .7314 L
s
.53234 .81009 m
.54055 .83631 L
s
.62487 .77315 m
.63292 .78492 L
s
.66128 .77688 m
.66922 .78708 L
s
.54395 .74499 m
.55218 .76193 L
s
.69771 .77919 m
.70554 .78848 L
s
.49113 .69782 m
.49946 .71176 L
s
.73412 .78015 m
.74184 .78902 L
s
.7705 .7802 m
.77813 .78917 L
s
.4878 .79117 m
.49591 .83096 L
s
.58034 .75515 m
.5885 .76809 L
s
.5206 .9152 m
.52889 .92378 L
s
.5357 .7311 m
.54395 .74499 L
s
.61681 .7624 m
.62487 .77315 L
s
.48271 .69013 m
.49113 .69782 L
s
.40675 .77616 m
.41351 .87266 L
s
.65332 .76733 m
.66128 .77688 L
s
.42937 .65816 m
.43825 .64978 L
s
.49591 .83096 m
.50408 .86691 L
s
.68985 .77033 m
.69771 .77919 L
s
.72636 .7716 m
.73412 .78015 L
s
.76285 .77154 m
.7705 .7802 L
s
.57215 .7442 m
.58034 .75515 L
s
.52741 .72082 m
.5357 .7311 L
s
.4921 .98025 m
.5007 .96313 L
s
.51231 .89581 m
.5206 .9152 L
s
.60871 .75289 m
.61681 .7624 L
s
.50408 .86691 m
.51231 .89581 L
s
.45113 .81574 m
.45893 .87453 L
s
.47417 .68858 m
.48271 .69013 L
s
.64533 .75852 m
.65332 .76733 L
s
.56394 .73548 m
.57215 .7442 L
s
.68196 .76189 m
.68985 .77033 L
s
.71857 .76335 m
.72636 .7716 L
s
.75516 .76319 m
.76285 .77154 L
s
.51906 .71433 m
.52741 .72082 L
s
.32815 .67221 m
.33996 .59742 L
s
.60059 .7447 m
.60871 .75289 L
s
.37465 .66314 m
.38469 .62534 L
s
.6373 .75044 m
.64533 .75852 L
s
.55568 .72898 m
.56394 .73548 L
s
.42023 .67428 m
.42937 .65816 L
s
.67403 .75384 m
.68196 .76189 L
s
.74743 .75508 m
.75516 .76319 L
s
.71075 .75533 m
.71857 .76335 L
s
.51064 .71137 m
.51906 .71433 L
s
.46549 .69237 m
.47417 .68858 L
s
.59243 .73776 m
.60059 .7447 L
s
.45893 .87453 m
.46693 .92463 L
s
.48355 .97989 m
.4921 .98025 L
s
.62924 .74301 m
.6373 .75044 L
s
.66606 .74608 m
.67403 .75384 L
s
.54737 .72444 m
.55568 .72898 L
s
.73967 .74714 m
.74743 .75508 L
s
.70289 .74746 m
.71075 .75533 L
s
.45423 1.06741 m
.46357 1.02041 L
s
.50213 .71127 m
.51064 .71137 L
s
.58423 .73185 m
.59243 .73776 L
s
.41351 .87266 m
.42063 .96092 L
s
.46693 .92463 m
.47514 .96091 L
s
.62113 .73607 m
.62924 .74301 L
s
.45667 .7001 m
.46549 .69237 L
s
.47514 .96091 m
.48355 .97989 L
s
.65806 .73849 m
.66606 .74608 L
s
.73187 .73925 m
.73967 .74714 L
s
.69498 .73961 m
.70289 .74746 L
s
.539 .72137 m
.54737 .72444 L
s
.41086 .69543 m
.42023 .67428 L
s
.57598 .72662 m
.58423 .73185 L
s
.61298 .72938 m
.62113 .73607 L
s
.49354 .71304 m
.50213 .71127 L
s
.65 .73092 m
.65806 .73849 L
s
.72402 .73129 m
.73187 .73925 L
s
.68703 .73165 m
.69498 .73961 L
s
.36426 .70661 m
.37465 .66314 L
s
.53057 .71909 m
.539 .72137 L
s
.44773 .70992 m
.45667 .7001 L
s
.56767 .72166 m
.57598 .72662 L
s
.60478 .72268 m
.61298 .72938 L
s
.6419 .72319 m
.65 .73092 L
s
.71613 .72314 m
.72402 .73129 L
s
.67902 .72348 m
.68703 .73165 L
s
.48486 .71547 m
.49354 .71304 L
s
.52208 .71683 m
.53057 .71909 L
s
.40131 .71824 m
.41086 .69543 L
s
.55931 .71648 m
.56767 .72166 L
s
.59653 .71571 m
.60478 .72268 L
s
.21054 .30307 m
.21886 .33088 L
s
.20057 .30327 m
.21054 .30307 L
s
.63375 .71515 m
.6419 .72319 L
s
.70818 .7147 m
.71613 .72314 L
s
.67096 .71497 m
.67902 .72348 L
s
.42063 .96092 m
.42824 1.0309 L
s
.31585 .75209 m
.32815 .67221 L
s
.21886 .33088 m
.22575 .38511 L
s
.44512 1.08675 m
.45423 1.06741 L
s
.43868 .71968 m
.44773 .70992 L
s
.70018 .7059 m
.70818 .7147 L
s
.66286 .70608 m
.67096 .71497 L
s
.62554 .70669 m
.63375 .71515 L
s
.47611 .71723 m
.48486 .71547 L
s
.58822 .70821 m
.59653 .71571 L
s
.55089 .71065 m
.55931 .71648 L
s
.51352 .71382 m
.52208 .71683 L
s
.18885 .33065 m
.20057 .30327 L
s
.22575 .38511 m
.23156 .46167 L
s
.69212 .69668 m
.70018 .7059 L
s
.6547 .69676 m
.66286 .70608 L
s
.35367 .74987 m
.36426 .70661 L
s
.61728 .69772 m
.62554 .70669 L
s
.57986 .7 m
.58822 .70821 L
s
.54241 .70378 m
.55089 .71065 L
s
.42824 1.0309 m
.43642 1.07444 L
s
.50491 .70935 m
.51352 .71382 L
s
.43642 1.07444 m
.44512 1.08675 L
s
.46731 .71708 m
.47611 .71723 L
s
.39167 .73906 m
.40131 .71824 L
s
.42957 .72719 m
.43868 .71968 L
s
.68401 .68705 m
.69212 .69668 L
s
.64649 .68702 m
.6547 .69676 L
s
.60897 .68821 m
.61728 .69772 L
s
.57145 .69096 m
.57986 .7 L
s
.53389 .6956 m
.54241 .70378 L
s
.49625 .70286 m
.50491 .70935 L
s
.63823 .67693 m
.64649 .68702 L
s
.67586 .67706 m
.68401 .68705 L
s
.60061 .6782 m
.60897 .68821 L
s
.45847 .71394 m
.46731 .71708 L
s
.56298 .68107 m
.57145 .69096 L
s
.52532 .68599 m
.53389 .6956 L
s
.42045 .7305 m
.42957 .72719 L
s
.23156 .46167 m
.23673 .55411 L
s
.48756 .69404 m
.49625 .70286 L
s
.17545 .38192 m
.18885 .33065 L
s
.62992 .66656 m
.63823 .67693 L
s
.66765 .66677 m
.67586 .67706 L
s
.5922 .66776 m
.60061 .6782 L
s
.55447 .6704 m
.56298 .68107 L
s
.38204 .75434 m
.39167 .73906 L
s
.51671 .67498 m
.52532 .68599 L
s
.25153 .42056 m
.25991 .4506 L
s
.2423 .40751 m
.25153 .42056 L
s
.44963 .70711 m
.45847 .71394 L
s
.25991 .4506 m
.2676 .49557 L
s
.62157 .65604 m
.62992 .66656 L
s
.6594 .65629 m
.66765 .66677 L
s
.47884 .68282 m
.48756 .69404 L
s
.58375 .65701 m
.5922 .66776 L
s
.34307 .78673 m
.35367 .74987 L
s
.54592 .65908 m
.55447 .6704 L
s
.30342 .8268 m
.31585 .75209 L
s
.23211 .41182 m
.2423 .40751 L
s
.50806 .66276 m
.51671 .67498 L
s
.41137 .72811 m
.42045 .7305 L
s
.44079 .69629 m
.44963 .70711 L
s
.2676 .49557 m
.27485 .55183 L
s
.47011 .66943 m
.47884 .68282 L
s
.61318 .64548 m
.62157 .65604 L
s
.6511 .64575 m
.6594 .65629 L
s
.57526 .64611 m
.58375 .65701 L
s
.53733 .64735 m
.54592 .65908 L
s
.49938 .64966 m
.50806 .66276 L
s
.43198 .68164 m
.44079 .69629 L
s
.46137 .65433 m
.47011 .66943 L
s
.60474 .63502 m
.61318 .64548 L
s
.64277 .63527 m
.6511 .64575 L
s
.56672 .63521 m
.57526 .64611 L
s
.37252 .7611 m
.38204 .75434 L
s
.52869 .63546 m
.53733 .64735 L
s
.40236 .71917 m
.41137 .72811 L
s
.49066 .6361 m
.49938 .64966 L
s
.22094 .43216 m
.23211 .41182 L
s
.4526 .63818 m
.46137 .65433 L
s
.4232 .66379 m
.43198 .68164 L
s
.27485 .55183 m
.28192 .6144 L
s
.29562 .49853 m
.30421 .52576 L
s
.2866 .48107 m
.29562 .49853 L
s
.30421 .52576 m
.31251 .56086 L
s
.23673 .55411 m
.24176 .65378 L
s
.59626 .62475 m
.60474 .63502 L
s
.55813 .62448 m
.56672 .63521 L
s
.63439 .62494 m
.64277 .63527 L
s
.52001 .62369 m
.52869 .63546 L
s
.48191 .62255 m
.49066 .6361 L
s
.41444 .64372 m
.4232 .66379 L
s
.44381 .62175 m
.4526 .63818 L
s
.39347 .70363 m
.40236 .71917 L
s
.31251 .56086 m
.32064 .60104 L
s
.27708 .47425 m
.2866 .48107 L
s
.40568 .62267 m
.41444 .64372 L
s
.43498 .60583 m
.44381 .62175 L
s
.47311 .60947 m
.48191 .62255 L
s
.3969 .60196 m
.40568 .62267 L
s
.38469 .68222 m
.39347 .70363 L
s
.51129 .61229 m
.52001 .62369 L
s
.16062 .45133 m
.17545 .38192 L
s
.5495 .61405 m
.55813 .62448 L
s
.33269 .81149 m
.34307 .78673 L
s
.58773 .61475 m
.59626 .62475 L
s
.62597 .61486 m
.63439 .62494 L
s
.36321 .75734 m
.37252 .7611 L
s
.35884 .59927 m
.36742 .62798 L
s
.35019 .57242 m
.35884 .59927 L
s
.34143 .5494 m
.35019 .57242 L
s
.32064 .60104 m
.32875 .64288 L
s
.36742 .62798 m
.37602 .65636 L
s
.37602 .65636 m
.38469 .68222 L
s
.38806 .58289 m
.3969 .60196 L
s
.33246 .53169 m
.34143 .5494 L
s
.42609 .59117 m
.43498 .60583 L
s
.26701 .47783 m
.27708 .47425 L
s
.46426 .59726 m
.47311 .60947 L
s
.37911 .56653 m
.38806 .58289 L
s
.50252 .60146 m
.51129 .61229 L
s
.32323 .52024 m
.33246 .53169 L
s
.28192 .6144 m
.28909 .67738 L
s
.54083 .604 m
.5495 .61405 L
s
.57917 .60506 m
.58773 .61475 L
s
.6175 .60507 m
.62597 .61486 L
s
.32875 .64288 m
.33698 .68265 L
s
.41712 .57834 m
.42609 .59117 L
s
.37003 .55366 m
.37911 .56653 L
s
.20887 .46559 m
.22094 .43216 L
s
.35417 .74235 m
.36321 .75734 L
s
.45535 .58622 m
.46426 .59726 L
s
.33698 .68265 m
.34544 .7168 L
s
.3137 .51526 m
.32323 .52024 L
s
.49369 .59134 m
.50252 .60146 L
s
.5321 .59438 m
.54083 .604 L
s
.57055 .59568 m
.57917 .60506 L
s
.609 .59558 m
.6175 .60507 L
s
.40807 .56771 m
.41712 .57834 L
s
.34544 .7168 m
.35417 .74235 L
s
.36079 .54466 m
.37003 .55366 L
s
.29131 .88582 m
.30342 .8268 L
s
.44636 .57652 m
.45535 .58622 L
s
.2564 .49048 m
.26701 .47783 L
s
.48481 .58197 m
.49369 .59134 L
s
.32274 .81977 m
.33269 .81149 L
s
.52333 .58516 m
.5321 .59438 L
s
.3989 .55942 m
.40807 .56771 L
s
.28909 .67738 m
.29661 .7345 L
s
.60045 .58635 m
.609 .59558 L
s
.56188 .58655 m
.57055 .59568 L
s
.30385 .51631 m
.3137 .51526 L
s
.35137 .53951 m
.36079 .54466 L
s
.24176 .65378 m
.24717 .7505 L
s
.4373 .56816 m
.44636 .57652 L
s
.47586 .57331 m
.48481 .58197 L
s
.5145 .57628 m
.52333 .58516 L
s
.59184 .57732 m
.60045 .58635 L
s
.38962 .55333 m
.3989 .55942 L
s
.55317 .5776 m
.56188 .58655 L
s
.34176 .53773 m
.35137 .53951 L
s
.42817 .56099 m
.4373 .56816 L
s
.29661 .7345 m
.30466 .77997 L
s
.29371 .52227 m
.30385 .51631 L
s
.24532 .50988 m
.2564 .49048 L
s
.46685 .56523 m
.47586 .57331 L
s
.31337 .80923 m
.32274 .81977 L
s
.19609 .50767 m
.20887 .46559 L
s
.50561 .56762 m
.5145 .57628 L
s
.58319 .56837 m
.59184 .57732 L
s
.5444 .5687 m
.55317 .5776 L
s
.38022 .54904 m
.38962 .55333 L
s
.30466 .77997 m
.31337 .80923 L
s
.14477 .53082 m
.16062 .45133 L
s
.41895 .55471 m
.42817 .56099 L
s
.33199 .53847 m
.34176 .53773 L
s
.45778 .55752 m
.46685 .56523 L
s
.49667 .55902 m
.50561 .56762 L
s
.57449 .55938 m
.58319 .56837 L
s
.53558 .55973 m
.5444 .5687 L
s
.37072 .54595 m
.38022 .54904 L
s
.28333 .53146 m
.29371 .52227 L
s
.40965 .54893 m
.41895 .55471 L
s
.27995 .91999 m
.29131 .88582 L
s
.24717 .7505 m
.25342 .83359 L
s
.44865 .54993 m
.45778 .55752 L
s
.48767 .55032 m
.49667 .55902 L
s
.56573 .55023 m
.57449 .55938 L
s
.5267 .55056 m
.53558 .55973 L
s
.2339 .53292 m
.24532 .50988 L
s
.32207 .5406 m
.33199 .53847 L
s
.36113 .54334 m
.37072 .54595 L
s
.40029 .54319 m
.40965 .54893 L
s
.43946 .54217 m
.44865 .54993 L
s
.47862 .54135 m
.48767 .55032 L
s
.55692 .54079 m
.56573 .55023 L
s
.51777 .54106 m
.5267 .55056 L
s
.27276 .54184 m
.28333 .53146 L
s
.18289 .5528 m
.19609 .50767 L
s
.31205 .54282 m
.32207 .5406 L
s
.54806 .53096 m
.55692 .54079 L
s
.5088 .53115 m
.51777 .54106 L
s
.46952 .53197 m
.47862 .54135 L
s
.43022 .53398 m
.43946 .54217 L
s
.39087 .53701 m
.40029 .54319 L
s
.35146 .5404 m
.36113 .54334 L
s
.25342 .83359 m
.26088 .89344 L
s
.26973 .92317 m
.27995 .91999 L
s
.53914 .52068 m
.54806 .53096 L
s
.49977 .52076 m
.5088 .53115 L
s
.46038 .52206 m
.46952 .53197 L
s
.2223 .55599 m
.2339 .53292 L
s
.42093 .52512 m
.43022 .53398 L
s
.3814 .52996 m
.39087 .53701 L
s
.34175 .53635 m
.35146 .5404 L
s
.26212 .55119 m
.27276 .54184 L
s
.30198 .5438 m
.31205 .54282 L
s
.53018 .50993 m
.53914 .52068 L
s
.49069 .50988 m
.49977 .52076 L
s
.45118 .51156 m
.46038 .52206 L
s
.4116 .51545 m
.42093 .52512 L
s
.3719 .5217 m
.3814 .52996 L
s
.26088 .89344 m
.26973 .92317 L
s
.12853 .61051 m
.14477 .53082 L
s
.33203 .53056 m
.34175 .53635 L
s
.48157 .49853 m
.49069 .50988 L
s
.52116 .4987 m
.53018 .50993 L
s
.44194 .50047 m
.45118 .51156 L
s
.40223 .50487 m
.4116 .51545 L
s
.29192 .54235 m
.30198 .5438 L
s
.36238 .51202 m
.3719 .5217 L
s
.2515 .55737 m
.26212 .55119 L
s
.16963 .59476 m
.18289 .5528 L
s
.4724 .48679 m
.48157 .49853 L
s
.5121 .48707 m
.52116 .4987 L
s
.21073 .57539 m
.2223 .55599 L
s
.43266 .48883 m
.44194 .50047 L
s
.32232 .52256 m
.33203 .53056 L
s
.39283 .4934 m
.40223 .50487 L
s
.35285 .50087 m
.36238 .51202 L
s
.28192 .53756 m
.29192 .54235 L
s
.46318 .47475 m
.4724 .48679 L
s
.50299 .47511 m
.5121 .48707 L
s
.42334 .47674 m
.43266 .48883 L
s
.3834 .48114 m
.39283 .4934 L
s
.31263 .51217 m
.32232 .52256 L
s
.34331 .48837 m
.35285 .50087 L
s
.24102 .55861 m
.2515 .55737 L
s
.45392 .46252 m
.46318 .47475 L
s
.49383 .46293 m
.50299 .47511 L
s
.41397 .46434 m
.42334 .47674 L
s
.27201 .52892 m
.28192 .53756 L
s
.37394 .46829 m
.3834 .48114 L
s
.30297 .49947 m
.31263 .51217 L
s
.33376 .47475 m
.34331 .48837 L
s
.19939 .58776 m
.21073 .57539 L
s
.44461 .45023 m
.45392 .46252 L
s
.48462 .45065 m
.49383 .46293 L
s
.40456 .45177 m
.41397 .46434 L
s
.36443 .45505 m
.37394 .46829 L
s
.29335 .4848 m
.30297 .49947 L
s
.26222 .51638 m
.27201 .52892 L
s
.23075 .55373 m
.24102 .55861 L
s
.32419 .4604 m
.33376 .47475 L
s
.43525 .43798 m
.44461 .45023 L
s
.47537 .43839 m
.48462 .45065 L
s
.39509 .43918 m
.40456 .45177 L
s
.15672 .62739 m
.16963 .59476 L
s
.35488 .4417 m
.36443 .45505 L
s
.28374 .46874 m
.29335 .4848 L
s
.25254 .50038 m
.26222 .51638 L
s
.31458 .44574 m
.32419 .4604 L
s
.42583 .42586 m
.43525 .43798 L
s
.46606 .42622 m
.47537 .43839 L
s
.38557 .42672 m
.39509 .43918 L
s
.11265 .67954 m
.12853 .61051 L
s
.22075 .54231 m
.23075 .55373 L
s
.27412 .45199 m
.28374 .46874 L
s
.34528 .42846 m
.35488 .4417 L
s
.24295 .48177 m
.25254 .50038 L
s
.18847 .59057 m
.19939 .58776 L
s
.30492 .43119 m
.31458 .44574 L
s
.26446 .4353 m
.27412 .45199 L
s
.2334 .4617 m
.24295 .48177 L
s
.41635 .41393 m
.42583 .42586 L
s
.45669 .41423 m
.46606 .42622 L
s
.37599 .41448 m
.38557 .42672 L
s
.21103 .52479 m
.22075 .54231 L
s
.33561 .41554 m
.34528 .42846 L
s
.29519 .41712 m
.30492 .43119 L
s
.22383 .44151 m
.2334 .4617 L
s
.25472 .41936 m
.26446 .4353 L
s
.20155 .50239 m
.21103 .52479 L
s
.05342 .4123 m
.05656 .51497 L
s
.21416 .42247 m
.22383 .44151 L
s
.40682 .40221 m
.41635 .41393 L
s
.44727 .40243 m
.45669 .41423 L
s
.36635 .40251 m
.37599 .41448 L
s
.32587 .40306 m
.33561 .41554 L
s
.19222 .47698 m
.20155 .50239 L
s
.28538 .40378 m
.29519 .41712 L
s
.17809 .58249 m
.18847 .59057 L
s
.24487 .40467 m
.25472 .41936 L
s
.18293 .45088 m
.19222 .47698 L
s
.17351 .42655 m
.18293 .45088 L
s
.20434 .4056 m
.21416 .42247 L
s
.16383 .40615 m
.17351 .42655 L
s
.14453 .64548 m
.15672 .62739 L
s
.08324 .40133 m
.09213 .43351 L
s
.13277 .43116 m
.14158 .46497 L
s
.12342 .40517 m
.13277 .43116 L
s
.39722 .39068 m
.40682 .40221 L
s
.35664 .39082 m
.36635 .40251 L
s
.4378 .39082 m
.44727 .40243 L
s
.31606 .39103 m
.32587 .40306 L
s
.27548 .39125 m
.28538 .40378 L
s
.09213 .43351 m
.09955 .48582 L
s
.23489 .3914 m
.24487 .40467 L
s
.11324 .39031 m
.12342 .40517 L
s
.19432 .39134 m
.20434 .4056 L
s
.15378 .39085 m
.16383 .40615 L
s
.14158 .46497 m
.15024 .50139 L
s
.16831 .56363 m
.17809 .58249 L
s
.15024 .50139 m
.15907 .53562 L
s
.14343 .37921 m
.15378 .39085 L
s
.38757 .3793 m
.39722 .39068 L
s
.34687 .37935 m
.35664 .39082 L
s
.42826 .37938 m
.4378 .39082 L
s
.30618 .37941 m
.31606 .39103 L
s
.15907 .53562 m
.16831 .56363 L
s
.18411 .37926 m
.19432 .39134 L
s
.26548 .37942 m
.27548 .39125 L
s
.22479 .37938 m
.23489 .3914 L
s
.09955 .48582 m
.10682 .54131 L
s
.17379 .36805 m
.18411 .37926 L
s
.2146 .36803 m
.22479 .37938 L
s
.25541 .36804 m
.26548 .37942 L
s
.29622 .36804 m
.30618 .37941 L
s
.33704 .36803 m
.34687 .37935 L
s
.37785 .36801 m
.38757 .3793 L
s
.41866 .36804 m
.42826 .37938 L
s
.13337 .64558 m
.14453 .64548 L
s
.09792 .72757 m
.11265 .67954 L
s
.10682 .54131 m
.11463 .5905 L
s
.20433 .35682 m
.2146 .36803 L
s
.24526 .3568 m
.25541 .36804 L
s
.2862 .35678 m
.29622 .36804 L
s
.32714 .35676 m
.33704 .36803 L
s
.36807 .35675 m
.37785 .36801 L
s
.40901 .35674 m
.41866 .36804 L
s
.05656 .51497 m
.06064 .60926 L
s
.12341 .62669 m
.13337 .64558 L
s
.23506 .34553 m
.24526 .3568 L
s
.27612 .3455 m
.2862 .35678 L
s
.31717 .34548 m
.32714 .35676 L
s
.35823 .34546 m
.36807 .35675 L
s
.39929 .34544 m
.40901 .35674 L
s
.11463 .5905 m
.12341 .62669 L
s
.26597 .33416 m
.27612 .3455 L
s
.30715 .33414 m
.31717 .34548 L
s
.34834 .33411 m
.35823 .34546 L
s
.38952 .33409 m
.39929 .34544 L
s
.29707 .32273 m
.30715 .33414 L
s
.33838 .32271 m
.34834 .33411 L
s
.37969 .32268 m
.38952 .33409 L
s
.06064 .60926 m
.06644 .68451 L
s
.08504 .74647 m
.09792 .72757 L
s
.32836 .31123 m
.33838 .32271 L
s
.36979 .3112 m
.37969 .32268 L
s
.35984 .29965 m
.36979 .3112 L
s
.06644 .68451 m
.0745 .73196 L
s
.0745 .73196 m
.08504 .74647 L
s
.25 Mabswid
.35855 0 m
.96518 .75898 L
s
.96518 .75898 m
1 1.28532 L
s
1 1.28532 m
.33887 .579 L
s
.33887 .579 m
.35855 0 L
s
.05012 .12232 m
0 .69467 L
s
0 .69467 m
.33887 .579 L
s
.33887 .579 m
.35855 0 L
s
.35855 0 m
.05012 .12232 L
s
.05012 .12232 m
.35855 0 L
s
.05725 .11949 m
.06336 .12629 L
s
[(0)] .04503 .1059 .89878 1 Mshowa
.12911 .091 m
.13512 .09788 L
s
[(0.25)] .11707 .07724 .87451 1 Mshowa
.202 .06209 m
.20792 .06905 L
s
[(0.5)] .19016 .04816 .85024 1 Mshowa
.27595 .03276 m
.28177 .0398 L
s
[(0.75)] .26431 .01866 .82597 1 Mshowa
.35099 .003 m
.3567 .01013 L
s
[(1)] .33955 -0.01126 .8017 1 Mshowa
.125 Mabswid
.07154 .11383 m
.07519 .11792 L
s
.08587 .10814 m
.08951 .11224 L
s
.10024 .10244 m
.10387 .10655 L
s
.11465 .09673 m
.11827 .10085 L
s
.1436 .08525 m
.1472 .08938 L
s
.15814 .07948 m
.16173 .08363 L
s
.17272 .0737 m
.17629 .07786 L
s
.18734 .0679 m
.1909 .07207 L
s
.21671 .05625 m
.22025 .06044 L
s
.23145 .05041 m
.23498 .0546 L
s
.24624 .04454 m
.24976 .04875 L
s
.26108 .03866 m
.26458 .04287 L
s
.29087 .02684 m
.29435 .03108 L
s
.30584 .02091 m
.3093 .02515 L
s
.32084 .01495 m
.3243 .01921 L
s
.33589 .00899 m
.33933 .01325 L
s
gsave
.13096 -0.02147 -70.2512 -16.5625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
69.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
0 .69467 m
.70973 1.3598 L
s
.02087 .71423 m
.02953 .7113 L
s
[(0)] .00356 .72009 1 -0.33867 Mshowa
.21712 .89815 m
.22584 .89541 L
s
[(1)] .19968 .90362 1 -0.31416 Mshowa
.39318 1.06315 m
.40195 1.06058 L
s
[(2)] .37564 1.06829 1 -0.29295 Mshowa
.55203 1.21201 m
.56084 1.20959 L
s
[(3)] .5344 1.21685 1 -0.27443 Mshowa
.69606 1.34699 m
.70491 1.34471 L
s
[(4)] .67836 1.35156 1 -0.25811 Mshowa
.125 Mabswid
.0619 .75268 m
.0671 .75095 L
s
.10201 .79027 m
.10722 .78856 L
s
.14123 .82703 m
.14645 .82534 L
s
.17959 .86298 m
.18481 .86131 L
s
.25384 .93256 m
.25908 .93094 L
s
.28979 .96625 m
.29503 .96465 L
s
.32498 .99923 m
.33023 .99764 L
s
.35943 1.03152 m
.36469 1.02996 L
s
.42625 1.09413 m
.43151 1.09261 L
s
.45864 1.12449 m
.46392 1.12299 L
s
.49039 1.15425 m
.49567 1.15276 L
s
.52151 1.18341 m
.5268 1.18195 L
s
.58195 1.24005 m
.58724 1.23862 L
s
.6113 1.26756 m
.6166 1.26614 L
s
.64009 1.29454 m
.64539 1.29314 L
s
.66834 1.32101 m
.67364 1.31963 L
s
gsave
.28794 1.09398 -71 -8.44114 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(t) show
69.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[8]:=",
  ImageSize->{211.75, 287.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40003C00017b000`400?l00000o`00003od`3oool0
0=<0oooo003C0?ooo`00d`3oool00=<0oooo003C0?ooo`00d`3oool00=<0oooo003C0?ooo`00d`3o
ool00=<0oooo000Y0?ooo`80000000<0oooo000000000000Y@3oool002X0oooo00<000000?ooo`00
00009P3oool5000007/0oooo000[0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool0
N`3oool002T0oooo0P0000000`3oool000000000000W0?ooo`030000003oool0oooo07/0oooo001E
0?ooo`030000003oool0oooo07/0oooo001E0?ooo`030000003oool0oooo07/0oooo001D0?ooo`80
0000O@3oool003D0oooo0P0000040?ooo`8000001@3oool00`000000oooo0?ooo`020?ooo`800000
RP3oool003@0oooo00@000000?ooo`3oool000002P3oool01@000000oooo0?ooo`3oool000000080
oooo00<000000?ooo`3oool0Q`3oool003@0oooo00@000000?ooo`3oool000002P3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo08L0oooo000d0?ooo`040000003oool0oooo000000/0
oooo00@000000?ooo`3oool0oooo0P00002:0?ooo`00=03oool010000000oooo0?ooo`0000080?oo
o`040000003oool0oooo000000<0oooo00<000000?ooo`3oool0403oool2000007L0oooo000]0?oo
o`800000103oool400000080oooo0P0000050?ooo`@000000`3oool3000000h0oooo0P0000000`3o
ool000000000001f0?ooo`00;03oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo
000001T0oooo0`0000020?ooo`80000000<0oooo0000003oool0M03oool002`0oooo00@000000?oo
o`3oool000002`3oool00`000000oooo0?ooo`0E0?ooo`8000000P3oool01`000000oooo0?ooo`3o
ool000000?ooo`000000M@3oool002`0oooo00@000000?ooo`3oool000002@3oool2000001D0oooo
0`0000000`3oool000000?ooo`050?ooo`040000003oool0oooo000007@0oooo000/0?ooo`040000
003oool0oooo000000T0oooo00<000000?ooo`3oool04P3oool2000000030?ooo`000000oooo00P0
oooo00D000000?ooo`3oool0oooo0000001c0?ooo`00;@3oool2000000X0oooo0`00000?0?ooo`<0
000000<0oooo0000003oool02P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0700
oooo000I0?ooo`800000103oool2000000<0oooo100000030?ooo`800000703oool3000000l0oooo
00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1_0?ooo`00603oool010000000oooo0?oo
o`0000080?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool00000603oool400000080
oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo06l0
oooo000H0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`0D0?ooo`<00000503oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo06h0
oooo000H0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3oool2000001@0
oooo0`0000020?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`1]0?ooo`00603oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo0000
00<0oooo00<000000?ooo`3oool04@3oool2000000<0oooo00<000000?ooo`3oool0503oool00`00
0000oooo0?ooo`060?ooo`030000003oool0oooo06`0oooo000I0?ooo`80000000<0oooo00000000
00001`3oool2000000@0oooo0`00000>0?ooo`<000000P3oool00`000000oooo0?ooo`0G0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool0J`3oool001/0oooo00@000000?ooo`3oool0
00006P3oool300000080oooo0P00000K0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3o
ool0J`3oool001/0oooo00@000000?ooo`3oool00000603oool200000080oooo00<000000?ooo`3o
ool0703oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo06X0oooo000K0?ooo`040000
003oool0oooo000001D0oooo0`0000020?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3o
ool02P3oool00`000000oooo0?ooo`1Y0?ooo`006`3oool010000000oooo0?ooo`00000C0?ooo`80
00000P3oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3o
ool0J03oool001`0oooo0P00000A0?ooo`<0000000<0oooo0000003oool09@3oool00`000000oooo
0?ooo`0<0?ooo`030000003oool0oooo06L0oooo000]0?ooo`80000000<0oooo0000003oool0:03o
ool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo06L0oooo000Z0?ooo`D000000P3oool0
0`000000oooo0?ooo`0V0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool0IP3oool0
02P0oooo1000000^0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0I@3oool002D0
oooo1000000a0?ooo`030000003oool0oooo00l0oooo00<000000?ooo`3oool0I03oool002<0oooo
1000000c0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0H`3oool00240oooo0P00
000g0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0H`3oool00240oooo1000000e
0?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0HP3oool00240oooo00<000000?oo
o`000000=P3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0640oooo000Q0?ooo`80
000000<0oooo0000003oool0=03oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo0600
oooo000I0?ooo`D000000`3oool01@000000oooo0?ooo`3oool0000003@0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`1P0?ooo`006`3oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool0<@3oool00`000000oooo0?ooo`0D0?ooo`030000
003oool0oooo05l0oooo000K0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`0a0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0GP3o
ool001<0oooo100000040?ooo`030000003oool0oooo0080oooo0P0000050?ooo`030000003oool0
oooo0300oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`1M0?ooo`006`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0;`3oool00`00
0000oooo0?ooo`0G0?ooo`030000003oool0oooo05`0oooo000J0?ooo`800000103oool00`000000
oooo0?ooo`060?ooo`030000003oool0oooo02h0oooo00<000000?ooo`3oool05`3oool00`000000
oooo0?ooo`1L0?ooo`00803oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo02d0oooo
00<000000?ooo`3oool0603oool00`000000oooo0?ooo`1K0?ooo`00803oool00`000000oooo0?oo
o`080?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?oo
o`1J0?ooo`00803oool2000000X0oooo00<000000?ooo`3oool0:`3oool00`000000oooo0?ooo`0J
0?ooo`030000003oool0oooo05T0oooo000P0?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool0:P3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo05P0oooo000P0?ooo`03
0000003oool0oooo00/0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0K0?ooo`03
0000003oool0oooo05P0oooo000P0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0
:03oool00`000000oooo0?ooo`0L0?ooo`030000003oool0oooo05L0oooo000P0?ooo`8000003@3o
ool00`000000oooo0?ooo`0X0?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3oool0EP3o
ool00200oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0W0?ooo`030000003oool0
oooo01h0oooo00<000000?ooo`3oool0E@3oool001l0oooo00<000000?ooo`3oool03`3oool00`00
0000oooo0?ooo`0U0?ooo`030000003oool0oooo0200oooo00<000000?ooo`3oool0E03oool001l0
oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0T0?ooo`030000003oool0oooo0200
oooo00<000000?ooo`3oool0E03oool001l0oooo00<000000?ooo`3oool04@3oool00`000000oooo
0?ooo`0S0?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool0D`3oool001l0oooo0P00
000C0?ooo`030000003oool0oooo0280oooo00<000000?ooo`3oool08P3oool00`000000oooo0?oo
o`1B0?ooo`007`3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo0240oooo00<00000
0?ooo`3oool08`3oool00`000000oooo0?ooo`1A0?ooo`007`3oool00`000000oooo0?ooo`0D0?oo
o`030000003oool0oooo0200oooo00<000000?ooo`3oool0903oool00`000000oooo0?ooo`1@0?oo
o`007`3oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo01l0oooo00<000000?ooo`3o
ool0903oool00`000000oooo0?ooo`1@0?ooo`007`3oool00`000000oooo0?ooo`0F0?ooo`030000
003oool0oooo01h0oooo00<000000?ooo`3oool09@3oool00`000000oooo0?ooo`1?0?ooo`007`3o
ool2000001L0oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0V0?ooo`030000003o
ool0oooo04h0oooo000O0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool07@3oool0
0`000000oooo0?ooo`0W0?ooo`030000003oool0oooo04d0oooo000<0?ooo`800000103oool20000
00@0oooo0P0000050?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool0703oool00`00
0000oooo0?ooo`0X0?ooo`030000003oool0oooo04`0oooo000;0?ooo`040000003oool0oooo0000
00P0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`0I0?ooo`030000003oool0
oooo01/0oooo00<000000?ooo`3oool0:03oool00`000000oooo0?ooo`1<0?ooo`002`3oool01000
0000oooo0?ooo`00000;0?ooo`050000003oool0oooo0?ooo`0000007@3oool00`000000oooo0?oo
o`0J0?ooo`030000003oool0oooo02T0oooo00<000000?ooo`3oool0B`3oool000D0oooo10000002
0?ooo`040000003oool0oooo000000T0oooo0P0000040?ooo`8000007@3oool00`000000oooo0?oo
o`0I0?ooo`030000003oool0oooo02X0oooo00<000000?ooo`3oool0BP3oool000/0oooo00@00000
0?ooo`3oool000002@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01d0oooo00<0
00000?ooo`3oool0603oool00`000000oooo0?ooo`0[0?ooo`030000003oool0oooo04T0oooo000<
0?ooo`8000002P3oool3000000<0oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0G
0?ooo`030000003oool0000002`0oooo00<000000?ooo`3oool0B03oool001h0oooo00<000000?oo
o`3oool0703oool4000001P0oooo00@000000?ooo`3oool00000:`3oool00`000000oooo0?ooo`18
0?ooo`007P3oool00`000000oooo0?ooo`0L0?ooo`050000003oool000000?ooo`0000005`3oool0
1@000000oooo0?ooo`3oool0000002/0oooo00<000000?ooo`3oool0A`3oool001h0oooo0P00000L
0?ooo`070000003oool0oooo0?ooo`000000oooo0000000B0?ooo`050000003oool0oooo0?ooo`00
0000103oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo04H0oooo000N0?ooo`030000
003oool0oooo01/0oooo00L000000?ooo`3oool0oooo0000003oool0000001<0oooo00@000000?oo
o`3oool00000103oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0oooo04D0oooo000N0?oo
o`030000003oool0oooo01X0oooo00<000000?ooo`3oool00`3oool00`000000oooo0000000;0?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`0000001@3oool00`000000oooo0?ooo`0Z0?oo
o`030000003oool0oooo04@0oooo000N0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3o
ool00`3oool010000000oooo0?ooo`00000;0?ooo`030000003oool0oooo00D0oooo0P0000050?oo
o`030000003oool0oooo02X0oooo00<000000?ooo`3oool0A03oool001h0oooo00<000000?ooo`3o
ool06P3oool00`000000oooo0?ooo`030?ooo`050000003oool0oooo0?ooo`0000000`3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`0Z0?ooo`030000003oool0oooo04<0oooo000N0?ooo`8000006P3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo0080oooo00D000000?ooo`3oool0oooo000000070?oo
o`030000003oool0oooo00@0oooo0P0000060?ooo`030000003oool0oooo02X0oooo00<000000?oo
o`3oool0@P3oool001d0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`040?ooo`04
0000003oool0oooo00000080oooo00D000000?ooo`3oool0oooo000000070?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`0000001P3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0
oooo0440oooo000M0?ooo`030000003oool0oooo01X0oooo00<000000?ooo`3oool0103oool01@00
0000oooo0?ooo`3oool000000080oooo00D000000?ooo`3oool0oooo000000070?ooo`030000003o
ool0oooo0080oooo00<000000?ooo`0000001`3oool00`000000oooo0?ooo`0Z0?ooo`030000003o
ool0oooo0400oooo000M0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00P3oool0
10000000oooo0?ooo`0000040?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool0
1@3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`0Y0?ooo`030000003oool0oooo0400oooo000M0?ooo`030000003oool0oooo01T0oooo
00<000000?ooo`3oool00`3oool010000000oooo0?ooo`0000030?ooo`050000003oool0oooo0?oo
o`0000000`3oool00`000000oooo0?ooo`050?ooo`030000003oool0000000@0oooo00<000000?oo
o`3oool01@3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo03l0oooo000M0?ooo`80
00006@3oool2000000H0oooo00<000000?ooo`000000103oool01@000000oooo0?ooo`3oool00000
00<0oooo00<000000?ooo`3oool01@3oool2000000D0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`0Y0?ooo`030000003oool0oooo03h0oooo000M0?ooo`030000003oool0oooo01T0
oooo00<000000?ooo`3oool01@3oool2000000D0oooo00@000000?ooo`3oool00000103oool00`00
0000oooo0?ooo`040?ooo`8000001P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
02T0oooo00<000000?ooo`3oool0?@3oool001d0oooo00<000000?ooo`3oool04`3oool00`000000
oooo0?ooo`020?ooo`030000003oool0000000L0oooo00<000000?ooo`3oool0103oool010000000
oooo0?ooo`0000040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo02P0oooo00<000000?ooo`3oool0?@3oool001d0
oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`020?ooo`040000003oool0oooo0000
00H0oooo0P0000060?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool00`3oool2
000000L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0
oooo03`0oooo000M0?ooo`8000003P3oool00`000000oooo0?ooo`040?ooo`050000003oool0oooo
0?ooo`0000000`3oool00`000000oooo0?ooo`030?ooo`030000003oool0000000H0oooo00@00000
0?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`030000003oool0000000L0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo03/0oooo000M
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0103oool010000000oooo0?ooo`00
00040?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000001`3oool010000000oooo0?oo
o`0000040?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0j0?ooo`00
7@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
103oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0080oooo00@000000?ooo`3o
ool000001P3oool01@000000oooo0?ooo`3oool0000000<0oooo00D000000?ooo`3oool0oooo0000
00020?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?oo
o`0Z0?ooo`030000003oool0oooo03T0oooo000E0?ooo`8000001P3oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool2000000H0oooo00D00000
0?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00D0oooo00D000000?ooo`3oool0oooo
000000030?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo02T0oooo00<000000?ooo`3oool0>@3oool001@0oooo
00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`8000001`3oool00`000000
oooo0?ooo`03000000030?ooo`000000oooo00H0oooo00D000000?ooo`3oool0oooo000000020?oo
o`040000003oool0oooo000000@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo02T0oooo00<000000?ooo`3oool0>03oool001@0oooo00@000000?oo
o`3oool00000103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00@00000
0?ooo`3oool000000`3oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo000000050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
1@3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo03L0oooo000D0?ooo`040000003o
ool0oooo000000@0oooo0P0000040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
1@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00H0oooo00<000000?ooo`000000
103oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo0?ooo`0000000P3oool2000000H0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo02T0
oooo00<000000?ooo`3oool0=P3oool001@0oooo00@000000?ooo`3oool00000103oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`030?ooo`8000001`3oool2000000030?ooo`000000oooo0080oooo0P0000070?ooo`05
0000003oool0oooo0?ooo`0000000`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo
03D0oooo000E0?ooo`8000001@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`030?ooo`800000203oool00`000000
oooo000000030?ooo`800000203oool01@000000oooo0?ooo`3oool000000080oooo0P0000060?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0Y0?oo
o`030000003oool0oooo03D0oooo00040?ooo`@00000503oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`04
0000003oool0oooo000000L0oooo00<000000?ooo`000000103oool2000000L0oooo00<000000?oo
o`3oool00P3oool010000000oooo0000000000070?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`3oool0103oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo03@0oooo00030?oo
o`040000003oool0oooo000001D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00P3oool010000000oooo0?ooo`00
00060?ooo`040000003oool00000000000@0oooo00<000000?ooo`0000001`3oool00`000000oooo
0?ooo`020?ooo`80000000<0oooo0000003oool01P3oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0c0?ooo`000`3o
ool010000000oooo0?ooo`00000E0?ooo`800000103oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool0000000D0
oooo00@000000?ooo`0000000000103oool00`000000oooo000000070?ooo`030000003oool0oooo
00<0oooo00@000000?ooo`3oool000001P3oool00`000000oooo0?ooo`060?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool0:P3oool00`000000oooo0?ooo`0b0?ooo`000P3oool20000
00030?ooo`000000000001D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0
oooo00<0oooo00@000000?ooo`0000000000103oool010000000oooo0?ooo`0000070?ooo`030000
003oool0oooo0080oooo0P0000020?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
1@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02X0oooo00<000000?ooo`3oool0
<@3oool001`0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo00H0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0oooo0080oooo
00D000000?ooo`3oool0oooo000000040?ooo`050000003oool0oooo0?ooo`0000001`3oool01`00
0000oooo0?ooo`3oool000000?ooo`0000000P3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0Y0?ooo`030000003o
ool0oooo0340oooo000L0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool01@3oool0
0`000000oooo0?ooo`050?ooo`050000003oool0oooo0?ooo`000000103oool01@000000oooo0?oo
o`3oool0000000<0oooo00<000000?ooo`3oool00`3oool010000000oooo0?ooo`0000080?ooo`04
0000003oool0oooo00000080oooo00<000000?ooo`0000001`3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0`0?oo
o`00703oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool01@3oool010000000oooo0?ooo`0000050?ooo`050000003oool0oooo0?ooo`0000000`3oool2
000000@0oooo00D000000?ooo`3oool0oooo000000070?ooo`040000003oool0oooo000000<0oooo
00<000000?ooo`0000001`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo
00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0_0?ooo`006`3oool2000000D0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`060?ooo`030000003oool0000000H0oooo00<0
00000?ooo`000000103oool2000000@0oooo00D000000?ooo`000000oooo000000080?ooo`030000
003oool0000000@0oooo0P0000080?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
1@3oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0oooo02h0oooo000K0?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0000000H0oooo00<000000?ooo`000000103oool00`000000oooo000000030?ooo`80000000@0
oooo0000000000000000203oool2000000@0oooo00<000000?ooo`0000001`3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0:P3oool00`000000oooo
0?ooo`0]0?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<0
00000?ooo`3oool01P3oool00`000000oooo000000060?ooo`8000001@3oool00`000000oooo0000
00030?ooo`8000000`3oool00`000000oooo0?ooo`060?ooo`8000000`3oool3000000030?ooo`00
0000oooo00H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo02T0oooo00<000000?ooo`3oool0;@3oool001/0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`050?ooo`030000003oool0oooo00H0oooo0P0000080?ooo`030000003oool0
oooo00<0oooo00P000000?ooo`3oool000000?ooo`000000oooo000000<0oooo0P0000080?ooo`06
0000003oool0oooo0?ooo`000000oooo0P0000000`3oool000000?ooo`050?ooo`030000003oool0
oooo00H0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0Y0?ooo`030000003oool0
oooo02`0oooo000K0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool01@3oool00`00
0000oooo0?ooo`070?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`0200000080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`060?oo
o`80000000T0oooo0000003oool0oooo0000003oool000000?ooo`0000001P3oool00`000000oooo
0?ooo`060?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0:@3oool00`000000oooo
0?ooo`0[0?ooo`006`3oool2000000D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?oo
o`060?ooo`030000003oool0oooo00D0oooo0P0000050?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo0?ooo`020?ooo`8000001`3oool010000000oooo0000000000020?ooo`03
0000003oool0oooo008000001`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`0Z0?ooo`006`3oool00`000000oooo
0?ooo`040?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`050?ooo`8000001@3oool020000000oooo0?ooo`000000oooo0000003oool000001@3oool2
000000D0oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`02000000L0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02X0oooo00<0
00000?ooo`3oool0:@3oool001/0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool01@3oool00`000000oooo00000005
0?ooo`070000003oool000000?ooo`000000oooo000000050?ooo`030000003oool0000000@0oooo
00<000000?ooo`3oool00P0000030?ooo`040000003oool0oooo0?ooo`8000001`3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0:@3oool00`000000
oooo0?ooo`0Y0?ooo`006`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00H0oooo
00<000000?ooo`3oool01@3oool2000000L0oooo00<000000?ooo`0000001@3oool2000000<0oooo
0P0000050?ooo`D000000P3oool01P000000oooo0?ooo`000000oooo00000080oooo00<000000?oo
o`3oool00P3oool2000000L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo02X0oooo00<000000?ooo`3oool0:03oool001/0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00D0oooo00<000000?ooo`00
00001@3oool01@000000oooo0?ooo`3oool0000000@0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool000000080oooo0P0000001@3oool000000?ooo`3oool0
00000080oooo00<000000?ooo`0000001@3oool2000000L0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo02X0oooo00<000000?ooo`3oool09`3oool0
01/0oooo0P0000050?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0103oool00`00
0000oooo000000050?ooo`050000003oool0oooo0?ooo`000000103oool00`000000oooo0?ooo`04
0?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0oooo0`0000000`3oool00000
0?ooo`030?ooo`<000001@3oool2000000H0oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`040?ooo`800000:`3oool00`000000oooo0?ooo`0V0?ooo`006P3oool2000000H0oooo00<0
00000?ooo`3oool01`3oool00`000000oooo0?ooo`040?ooo`030000003oool0000000D0oooo00<0
00000?ooo`3oool00P3oool00`000000oooo0?ooo`02000000H0oooo00<000000?ooo`0000000`3o
ool00`000000oooo0?ooo`020?ooo`@000001P3oool00`000000oooo0?ooo`040?ooo`8000001P3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0:@3o
ool00`000000oooo0?ooo`0U0?ooo`001`3oool2000000@0oooo0P0000040?ooo`8000001@3oool0
0`000000oooo0?ooo`050?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0103oool0
0`000000oooo000000050?ooo`030000003oool0oooo0080oooo00D000000?ooo`000000oooo0000
00060?ooo`800000103oool00`000000oooo0?ooo`06000000H0oooo0P0000050?ooo`030000003o
ool0000000H0oooo0P0000060?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0:03o
ool00`000000oooo0?ooo`0U0?ooo`001P3oool010000000oooo0?ooo`0000080?ooo`040000003o
ool0oooo000000@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`070?ooo`030000
003oool0oooo00<0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0oooo00@0
oooo0P0000000`3oool000000?ooo`050?ooo`030000003oool0000000H000000`3oool00`000000
oooo0000000400000080oooo00<000000?ooo`0000001@3oool00`000000oooo000000070?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0Y0?ooo`03
0000003oool0oooo02@0oooo00060?ooo`040000003oool0oooo000000/0oooo00<000000?ooo`3o
ool00P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3o
ool00`3oool01@000000oooo0?ooo`3oool0000000<0oooo00<000000?ooo`3oool0103oool20000
00030?ooo`000000oooo00<0oooo0`0000060?ooo`030000003oool0oooo0080oooo00<000000?oo
o`0000000P3oool00`000000oooo0000000200000080oooo0P0000040?ooo`030000003oool00000
00L0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
02T0oooo00<000000?ooo`3oool08`3oool000H0oooo00@000000?ooo`3oool000002@3oool20000
00D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo
00<0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003oool0oooo00@0oooo00@00000
0?ooo`00000000000P3oool200000080oooo00<000000?ooo`3oool0103oool00`000000oooo0?oo
o`020?ooo`800000103oool010000000oooo0?ooo`0000040?ooo`8000000`3oool010000000oooo
0000000000050?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`0Y0?ooo`030000003oool0oooo0280oooo00060?ooo`040000003oool0oooo000000T0
oooo00<000000?ooo`3oool0103oool2000000H0oooo00<000000?ooo`3oool01`3oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool000001@3oool01000
0000oooo0?ooo`3oool3000000@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`02
0?ooo`800000103oool00`000000oooo0?ooo`03000000@0oooo0P0000020?ooo`040000003oool0
oooo000000D0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo02T0oooo00<000000?ooo`3oool08@3oool000L0oooo0P00000:0?ooo`<00000103oool0
0`000000oooo0?ooo`060?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0000000H0oooo00D000000?ooo`3oool0oooo0000
00060?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00P3oool2000000D0oooo00<0
00000?ooo`0000000P3oool2000000@0oooo00@000000?ooo`3oool000000P3oool00`000000oooo
0?ooo`030?ooo`8000001@3oool00`000000oooo0?ooo`050?ooo`800000:@3oool00`000000oooo
0?ooo`0Q0?ooo`006P3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00L0oooo00D0
00000?ooo`3oool0oooo000000060?ooo`8000001@3oool00`000000oooo0?ooo`020?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`020?ooo`<00000
1@3oool00`000000oooo000000030?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool0
00000P3oool2000000D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo02P0oooo00<000000?ooo`3oool0803oool001X0oooo00<000000?ooo`3oool0
1P3oool00`000000oooo0?ooo`070?ooo`050000003oool0oooo0?ooo`0000001P3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`050?ooo`030000003oool0oooo00@000001@3oool00`000000oooo000000040?ooo`03
0000003oool0oooo0080oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00<0oooo0P0000060?ooo`030000003oool0oooo02P0oooo00<000000?oo
o`3oool07`3oool001X0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`070?ooo`05
0000003oool0oooo0?ooo`0000001P3oool2000000@0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo0`0000020?ooo`030000003oool00000
00D0oooo0P0000040?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool000000P3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`0X0?ooo`030000003oool0oooo01h0oooo000J0?ooo`030000003oool0oooo
00H0oooo00<000000?ooo`3oool01`3oool01@000000oooo0?ooo`3oool0000000D0oooo00@00000
0?ooo`3oool000000P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00@0
00000?ooo`3oool0oooo100000030?ooo`040000003oool00000000000D0oooo0P0000001P3oool0
0000000000000000oooo000000D0oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?oo
o`0X0?ooo`030000003oool0oooo01d0oooo000I0?ooo`800000203oool00`000000oooo0?ooo`07
0?ooo`050000003oool0oooo0?ooo`0000001@3oool010000000oooo0?ooo`3oool2000000L0oooo
00<000000?ooo`3oool01@3oool4000000<0oooo00D000000?ooo`3oool0oooo000000020?ooo`80
00001@3oool200000080oooo00<000000?ooo`0000001@3oool010000000oooo0?ooo`0000030?oo
o`030000003oool0oooo00<0oooo0P0000050?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool0:03oool00`000000oooo0?ooo`0M0?ooo`006P3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo00L0oooo00@000000?ooo`3oool000001P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo00@0oooo0P0000070?ooo`040000003oool0oooo000000<0oooo00<00000
0?ooo`3oool00`3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool00`3o
ool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02P0oooo00<000000?ooo`3oool0
703oool001T0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`080?ooo`030000003o
ool0000000D0oooo00<000000?ooo`3oool02`3oool010000000oooo0?ooo`3oool3000000030?oo
o`000000oooo00H0oooo00@000000?ooo`3oool000000`3oool2000000@0oooo0P0000030?ooo`04
0000003oool0oooo000000D0oooo00@000000?ooo`3oool000000P3oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo00<0oooo0P0000050?ooo`800000:@3oool00`000000oooo0?ooo`0K
0?ooo`006@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`0000001@3oool00`000000oooo0?ooo`0<0?ooo`<00000103oool00`000000oooo0?ooo`050?oo
o`040000003oool0oooo000000<0oooo00<000000?ooo`0000000`3oool2000000<0oooo00D00000
0?ooo`3oool0oooo000000050?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
9`3oool00`000000oooo0?ooo`0J0?ooo`006@3oool00`000000oooo0?ooo`070?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`0000001@3oool00`000000oooo0?ooo`090?ooo`@000001P3o
ool00`000000oooo0?ooo`050?ooo`040000003oool0oooo000000<0oooo00D000000?ooo`3oool0
00000?ooo`03000000<0oooo00D000000?ooo`3oool0oooo000000060?ooo`040000003oool0oooo
00000080oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo00D0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`0I0?ooo`006@3oool20000
00P0oooo00<000000?ooo`3oool02@3oool2000000@0oooo00<000000?ooo`3oool01`3oool30000
00<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`050?ooo`030000003oool00000
00D0oooo00@000000?ooo`000000oooo0P0000000`3oool000000?ooo`020?ooo`050000003oool0
oooo0?ooo`0000001@3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
9`3oool00`000000oooo0?ooo`0I0?ooo`006@3oool00`000000oooo0?ooo`070?ooo`030000003o
ool0oooo00T0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`040?ooo`<000001P3o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00H0oooo0P0000050?ooo`030000003o
ool0oooo00<0000000<0oooo0000003oool00P3oool01@000000oooo0?ooo`3oool0000000H0oooo
00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3o
ool0603oool001T0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`090?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool00P3oool3000000T0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`060?ooo`8000001@3oool010000000oooo0?ooo`3oool200000003
0?ooo`000000oooo0080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`0000001P3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?ooo`0G0?ooo`00
6@3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00T0oooo0P0000030?ooo`030000
003oool0oooo00800000303oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo
0P0000050?ooo`060000003oool0oooo0000003oool000000P3oool010000000oooo0?ooo`000004
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00P3oool00`000000oooo00000006
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0W
0?ooo`030000003oool0oooo01H0oooo000I0?ooo`030000003oool0oooo00L0oooo00<000000?oo
o`3oool02@3oool200000080oooo1000000>0?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool01@3oool2000000H0oooo00@000000?ooo`000000oooo0P0000000`3oool000000?ooo`02
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`02
0?ooo`030000003oool0000000H0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool05P3oool001T0oooo00<000000?oo
o`3oool01`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00<000004@3oool00`00
0000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo0P0000060?ooo`030000003oool00000
0080oooo0P0000001@3oool000000?ooo`3oool0000000@0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`020?ooo`040000003oool0oooo000000H0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool05@3o
ool001T0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`070?ooo`<00000503oool0
0`000000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo0P0000070?ooo`8000000P3oool0
20000000oooo0000003oool000000?ooo`000000oooo0P0000020?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool00`0000020?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3o
ool0503oool001T0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`040?ooo`@00000
5P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00D0oooo0P0000070?ooo`050000
003oool0oooo0?ooo`0000000P3oool2000000030?ooo`00000000000080oooo00<000000?ooo`00
00001`3oool01P000000oooo000000000000oooo00000080oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool09`3oool0
0`000000oooo0?ooo`0C0?ooo`006@3oool00`000000oooo0?ooo`080?ooo`040000003oool0oooo
0?ooo`<000000`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`3oool01@3oool2000000L0oooo00D000000?ooo`3oool0oooo000000030?ooo`<00000103o
ool00`000000oooo000000070?ooo`80000000H0oooo0000003oool000000?ooo`0000060?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0X0?ooo`03
0000003oool0oooo0180oooo000H0?ooo`030000003oool0oooo00T0oooo100000050?ooo`030000
003oool0oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`040?ooo`800000
1`3oool200000080oooo00<000000?ooo`3oool00P3oool2000000@0oooo00<000000?ooo`000000
1`3oool010000000oooo0?ooo`0000020?ooo`030000003oool0000000H0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3o
ool04P3oool001P0oooo00<000000?ooo`3oool0203oool2000000P0oooo00<000000?ooo`3oool0
5@3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00@0oooo0P0000060?ooo`030000
003oool000000080oooo00@000000?ooo`3oool0oooo100000030?ooo`040000003oool0oooo0000
00H0oooo0P0000020?ooo`060000003oool0oooo0000003oool000001P3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool09`3oool00`000000oooo0?oo
o`0A0?ooo`00603oool3000000D0oooo0`0000000`3oool000000?ooo`070?ooo`030000003oool0
oooo01D0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`040?ooo`030000003oool0
000000D0oooo00H000000?ooo`3oool000000?ooo`0000020?ooo`030000003oool0oooo00800000
103oool00`000000oooo000000060?ooo`050000003oool000000?ooo`0000000`3oool00`000000
oooo000000060?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`0W0?ooo`030000003oool0oooo0100oooo000@0?ooo`D000000`3oool00`000000oooo
0?ooo`020?ooo`<00000103oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo01D0oooo
00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`040?ooo`030000003oool0000000D0oooo
00H000000?ooo`3oool000000?ooo`0000020?ooo`030000003oool0oooo00<000000`3oool01000
0000oooo0?ooo`0000050?ooo`030000003oool0oooo00800000103oool2000000H0oooo00<00000
0?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02P0oooo00<00000
0?ooo`3oool03`3oool00180oooo00<000000?ooo`3oool00`3oool00`000000oooo000000020000
00L0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo
00H0oooo00<000000?ooo`3oool0103oool00`000000oooo000000050?ooo`080000003oool0oooo
0000003oool000000?ooo`0000040?ooo`030000003oool000000080oooo00<000000?ooo`000000
0P0000040?ooo`030000003oool0oooo008000001@3oool2000000H0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02P0oooo00<000000?ooo`3oool0
3P3oool00180oooo00<000000?ooo`3oool00`3oool2000000X0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00`3o
ool010000000oooo0?ooo`0000040?ooo`090000003oool0oooo0?ooo`000000oooo0000003oool0
000000@0oooo0`0000030?ooo`030000003oool0oooo008000000P3oool00`000000oooo0?ooo`02
0?ooo`030000003oool0oooo00@0oooo0P0000060?ooo`030000003oool0oooo00@0oooo0P000005
0?ooo`800000:03oool00`000000oooo0?ooo`0>0?ooo`004P3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00T0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0F0?oo
o`030000003oool0oooo00H0oooo00<000000?ooo`3oool00`3oool010000000oooo0?ooo`000004
0?ooo`030000003oool0oooo0080oooo0P0000000`3oool000000?ooo`030?ooo`030000003oool0
000000@000000`3oool010000000oooo0?ooo`0000040?ooo`8000001P3oool2000000H0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02L0oooo00<0
00000?ooo`3oool03@3oool00180oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo00<0oooo00D000000?ooo`3oool0oooo000000030?ooo`030000003o
ool0oooo0080oooo0`0000050?ooo`030000003oool0oooo0080000000<0oooo0000003oool00`3o
ool00`000000oooo000000050?ooo`8000001@3oool2000000L0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool0303o
ool00140oooo0P0000080?ooo`030000003oool0000000L0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0H0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00P3oool0
1@000000oooo0?ooo`3oool0000000<0oooo00<000000?ooo`3oool00`3oool2000000D0oooo00P0
00000?ooo`00000000000?ooo`000000oooo000000<0oooo100000040?ooo`040000003oool00000
000000@0oooo0P0000070?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`0W0?ooo`030000003oool0oooo00/0oooo000L0?ooo`030000003oool0oooo
00H0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo
00H0oooo00<000000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool000000080oooo00<00000
0?ooo`3oool0103oool2000000D0oooo0P0000020?ooo`L0000000<0oooo0000000000000`000002
0?ooo`050000003oool0oooo0?ooo`000000103oool3000000H0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02L0oooo00<000000?ooo`3oool02P3o
ool001@0oooo0P0000070?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo0000
000K0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00P3oool01@000000oooo0?oo
o`3oool000000080oooo00<000000?ooo`3oool0103oool2000000D0oooo0P0000050?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool00P3oool3000000@0oooo0P0000030?ooo`040000
003oool00000000000D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo02H0oooo00<000000?ooo`3oool02P3oool001<0oooo00@000000?ooo`3oool0
00001`3oool00`000000oooo0?ooo`050?ooo`050000003oool0oooo0?ooo`0000006`3oool00`00
0000oooo0?ooo`060?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool00`00
0000oooo000000060?ooo`<000000`3oool3000000D0oooo0P0000030?ooo`800000103oool50000
00<0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02H0oooo00<00000
0?ooo`3oool02@3oool001<0oooo00@000000?ooo`3oool00000203oool00`000000oooo0?ooo`04
0?ooo`040000003oool0oooo000001`0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?oo
o`020?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000001@3oool010000000oooo0000
003oool300000080oooo00<000000?ooo`3oool00`3oool00`000000oooo000000020?ooo`800000
1@3oool01@000000oooo0?ooo`3oool0000000<0oooo0P0000020?ooo`040000003oool0oooo0000
00D0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
02H0oooo00<000000?ooo`3oool0203oool001<0oooo00@000000?ooo`3oool000002@3oool20000
00D0oooo00<000000?ooo`000000703oool00`000000oooo0?ooo`060?ooo`050000003oool0oooo
0?ooo`0000001P3oool00`000000oooo0?ooo`040?ooo`030000003oool0000000H0oooo00<00000
0?ooo`3oool0103oool010000000oooo0000003oool2000000D0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`020?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
9P3oool00`000000oooo0?ooo`070?ooo`004`3oool010000000oooo0?ooo`00000;0?ooo`030000
003oool0oooo0080oooo0P00000M0?ooo`030000003oool0oooo00H0oooo00D000000?ooo`3oool0
oooo000000060?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool000001P3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo0080000000<0oooo0000003oool00`3oool20000
00@0oooo00<000000?ooo`3oool00P3oool2000000050?ooo`000000oooo0?ooo`0000001@3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool09P3oool0
0`000000oooo0?ooo`060?ooo`00503oool2000000d0oooo0P0000030?ooo`030000003oool0oooo
01/0oooo00<000000?ooo`3oool01`3oool010000000oooo0?ooo`0000060?ooo`800000103oool0
10000000oooo0?ooo`0000060?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00P00
00000`3oool000000?ooo`040?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool0
0P3oool010000000oooo0?ooo`0000020?ooo`800000103oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool09@3oool00`000000oooo0?ooo`060?ooo`00
8`3oool200000240oooo00<000000?ooo`3oool01`3oool010000000oooo0?ooo`0000060?ooo`80
00000`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool0103oool020000000oooo0?ooo`000000oooo0000003oool00000103oool4000000<0oooo00<0
00000?ooo`3oool00P3oool010000000oooo0?ooo`0000020?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo02D0oooo
00<000000?ooo`3oool01@3oool002D0oooo00<000000?ooo`3oool07P3oool00`000000oooo0?oo
o`070?ooo`040000003oool0oooo000000D0oooo00<000000?ooo`3oool00`0000050?ooo`030000
003oool0oooo00<0oooo00<000000?ooo`3oool0103oool010000000oooo0?ooo`0000020?ooo`80
0000103oool200000080oooo00D000000?ooo`3oool0oooo000000040?ooo`030000003oool00000
00<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool0903oool2000000H0oooo000V0?ooo`030000003oool0oooo01d0
oooo00<000000?ooo`3oool01`3oool010000000oooo0?ooo`0000050?ooo`030000003oool0oooo
00P0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`050?ooo`030000003oool00000
00<0oooo0P0000030?ooo`050000003oool000000?ooo`0000000`3oool00`000000oooo0?ooo`03
0?ooo`030000003oool0000000<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0803oool3000000030?ooo`000000
oooo00D0oooo000W0?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool0203oool20000
00H0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
00D0oooo00<000000?ooo`000000103oool01`000000oooo0?ooo`3oool000000?ooo`0000000P3o
ool01@000000oooo0?ooo`3oool0000000@0oooo00@000000?ooo`3oool000000`3oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool00`000000
oooo0?ooo`0K0?ooo`@00000103oool00`000000oooo0?ooo`040?ooo`00:03oool00`000000oooo
0?ooo`0K0?ooo`030000003oool0oooo00P0oooo0P0000060?ooo`030000003oool0oooo00P0oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0000000@0oooo
0P0000020?ooo`060000003oool0oooo0000003oool000000`3oool00`000000oooo0?ooo`030?oo
o`040000003oool0oooo000000<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0603oool3000000P0oooo00<00000
0?ooo`3oool0103oool002T0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`080?oo
o`8000001@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool0103oool2000000H0oooo00<000000?ooo`3oool00P0000001@3oool000000?ooo`3oool0
000000<0oooo00<000000?ooo`3oool00`3oool010000000oooo0?ooo`0000030?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo01@0oooo0`00000;0?ooo`030000003oool0oooo00@0oooo000Z0?ooo`030000003oool0
oooo01T0oooo00<000000?ooo`3oool0203oool2000000D0oooo00<000000?ooo`3oool02@3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0103oool3
000000040?ooo`000000oooo00000080oooo00<000000?ooo`3oool00P3oool00`000000oooo0?oo
o`030?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`030?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool03`3oool4000000h0oooo
00<000000?ooo`3oool0103oool002/0oooo0P00000I0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0103oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00@0oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`050000003oool000000?ooo`000000
0P3oool00`000000oooo000000040?ooo`030000003oool0oooo00<0oooo00D000000?ooo`3oool0
oooo000000020?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo00/0oooo0`00000B0?ooo`030000003oool0oooo00@0
oooo000/0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool01`3oool2000000@0oooo
00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo
00<000000?ooo`3oool0103oool00`000000oooo0000000200000080oooo00<000000?ooo`000000
1@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo008000000P3oool00`000000oooo
0?ooo`040?ooo`030000003oool0oooo00<0oooo0P0000050?ooo`030000003oool0oooo00L0oooo
0`00000E0?ooo`030000003oool0oooo00@0oooo000]0?ooo`030000003oool0oooo01L0oooo00<0
00000?ooo`3oool01`3oool2000000@0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool01@3oool010000000oooo0?oo
o`0000030?ooo`8000001@3oool00`000000oooo0?ooo`040?ooo`030000003oool0000000800000
0P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`020?ooo`@00000603oool00`000000oooo0?ooo`040?ooo`00;P3o
ool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00L0oooo0P0000030?ooo`030000003o
ool0oooo00/0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool00P0000020?ooo`8000001P3oool00`000000oooo0?oo
o`030?ooo`80000000@0oooo0000003oool000000P3oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo00@0oooo00<000000?ooo`3oool0103oool4000001d0oooo00<000000?ooo`3oool0
0`3oool002l0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`070?ooo`030000003o
ool000000080oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`050?ooo`030000003o
ool0oooo00@0oooo0P0000060?ooo`090000003oool0oooo0000003oool000000?ooo`000000oooo
00<000000`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo008000000P3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00P3oool30000
0200oooo00<000000?ooo`3oool00`3oool00300oooo0P00000E0?ooo`030000003oool0oooo00H0
oooo00@000000?ooo`3oool0oooo0P00000>0?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool0103oool2000000H0oooo00@000000?ooo`3oool000000P3oool010000000oooo00000000
00020?ooo`8000000P3oool00`000000oooo0?ooo`030?ooo`060000003oool0oooo0000003oool0
00000P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo100000030?ooo`03
0000003oool0oooo01d0oooo00<000000?ooo`3oool00`3oool00380oooo0P00000C0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`050?ooo`030000
003oool0oooo00@0oooo0P0000060?ooo`80000000<0oooo0000003oool00P3oool3000000<0oooo
00@000000?ooo`3oool000001@3oool2000000070?ooo`000000oooo0?ooo`000000oooo00000006
0?ooo`030000003oool0oooo00<0000000<0oooo0000003oool01@3oool00`000000oooo0?ooo`0L
0?ooo`030000003oool0oooo00<0oooo000b0?ooo`8000004`3oool00`000000oooo0?ooo`060?oo
o`030000003oool0oooo0180oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?oo
o`8000001@3oool01@000000oooo0000003oool0000000@0oooo0P0000040?ooo`040000003oool0
oooo000000@0oooo00@000000?ooo`00000000000`3oool00`000000oooo000000050?ooo`<00000
1@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01/0oooo00<000000?ooo`3oool0
0`3oool003@0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`8000001@3o
ool00`000000oooo0?ooo`02000000@0oooo0P0000040?ooo`040000003oool0oooo000000<0oooo
00@000000?ooo`3oool0oooo0P0000030?ooo`030000003oool0000000@000000P3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool06P3oool00`000000
oooo0?ooo`030?ooo`00=@3oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo00H0oooo
00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo
00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`02000000<0oooo00@000000?ooo`00
000000000`3oool01@000000oooo0?ooo`3oool000000080oooo00@000000?ooo`3oool0oooo0P00
00040?ooo`<000001P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<0
00000?ooo`3oool06P3oool00`000000oooo0?ooo`030?ooo`00=P3oool00`000000oooo0?ooo`0>
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool00000103oool01P000000oooo0?oo
o`000000oooo00000080oooo00<000000?ooo`3oool00P0000030?ooo`070000003oool0oooo0?oo
o`000000oooo000000040?ooo`030000003oool0oooo0080oooo00<000000?ooo`0000001`3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool06@3oool0
0`000000oooo0?ooo`030?ooo`00=`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo
00D0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
0080oooo00@000000?ooo`3oool00000103oool010000000oooo0?ooo`3oool200000080oooo00<0
00000?ooo`3oool00`0000020?ooo`D0000000<0oooo0000003oool00`3oool2000000<0oooo00@0
00000?ooo`3oool000001P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo
00<000000?ooo`3oool0603oool00`000000oooo0?ooo`030?ooo`00>03oool00`000000oooo0?oo
o`0<0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?oo
o`050?ooo`030000003oool0oooo0080oooo00@000000?ooo`3oool00000103oool010000000oooo
0?ooo`3oool2000000030?ooo`000000oooo0080oooo0`0000020?ooo`8000000`3oool2000000@0
oooo00<000000?ooo`0000000P3oool01@000000oooo0?ooo`3oool0000000H0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01L0oooo00<000000?oo
o`3oool00`3oool003T0oooo0P00000<0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool04`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo0080oooo00D000000?ooo`3o
ool0oooo000000020?ooo`030000003oool0oooo0080oooo0P0000000`3oool000000?ooo`020?oo
o`030000003oool000000080000000<0oooo0000003oool00`3oool00`000000oooo0?ooo`020?oo
o`060000003oool0oooo0000003oool00000103oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0F0?ooo`030000003o
ool0oooo00<0oooo000j0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`0C0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00P3oool0
1@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool00`3oool2000000@0oooo00<0
00000?ooo`3oool00P0000000`3oool000000?ooo`030?ooo`030000003oool0oooo0080oooo00@0
00000?ooo`3oool0oooo0P0000050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo01H0oooo00<000000?ooo`3oool0
0`3oool003/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo01<0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`050000003o
ool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`030?ooo`8000001@3oool00`000000oooo
00000003000000@0oooo100000000`3oool000000?ooo`030?ooo`030000003oool0oooo00@0oooo
0P0000050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`0E0?ooo`030000003oool0oooo00<0oooo000l0?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00D0oooo00<0
00000?ooo`3oool00P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0
0`3oool2000000D0oooo0P0000020?ooo`P000000P3oool3000000<0oooo0`0000060?ooo`030000
003oool0oooo00<0oooo0P0000050?ooo`030000003oool0oooo00<0oooo0P00000E0?ooo`030000
003oool0oooo00<0oooo000m0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool00P3o
ool01`000000oooo0?ooo`3oool000000?ooo`0000001P3oool2000000D0oooo00<000000?ooo`3o
ool00`3oool010000000oooo0?ooo`3oool2000000@0oooo1@0000020?ooo`800000103oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`0C0?ooo`030000003oool0oooo00<0oooo000n0?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`8000001P3oool00`000000
oooo000000040?ooo`030000003oool0oooo00<0oooo0P0000001@3oool000000?ooo`3oool00000
00<0oooo0P0000020?ooo`8000000`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
0180oooo00<000000?ooo`3oool00`3oool003l0oooo00@000000?ooo`3oool000001@3oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo01@0oooo00<000000?ooo`3oool01P3oool01@00
0000oooo0?ooo`3oool0000000@0oooo0P0000060?ooo`030000003oool0oooo00D000001@3oool0
10000000oooo0000000000020?ooo`0;0000003oool0oooo0?ooo`000000oooo0000003oool00000
0?ooo`0000000`3oool00`000000oooo0?ooo`020?ooo`8000001@3oool00`000000oooo0?ooo`03
0?ooo`8000001@3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo0080oooo00100?oo
o`8000001P3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo01D0oooo00<000000?oo
o`3oool01P3oool010000000oooo0?ooo`0000050?ooo`8000001@3oool2000000H0oooo00<00000
0?ooo`3oool0103oool00`000000oooo000000030?ooo`050000003oool0oooo0000003oool00`00
00020?ooo`040000003oool0oooo0?ooo`800000103oool00`000000oooo0?ooo`030?ooo`800000
103oool00`000000oooo0?ooo`040?ooo`8000004P3oool00`000000oooo0?ooo`020?ooo`00>03o
ool5000000@0oooo0P0000060?ooo`050000003oool0oooo0?ooo`0000005`3oool00`000000oooo
0?ooo`060?ooo`040000003oool0oooo000000H0oooo00<000000?ooo`3oool00`3oool2000000H0
oooo00<000000?ooo`3oool0103oool010000000oooo0000000000020?ooo`050000003oool0oooo
0000003oool00`0000030?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`0@0?ooo`030000003oool0oooo0080oooo000j0?ooo`030000003oool0oooo
00H0oooo00<000000?ooo`3oool00`3oool01@000000oooo0?ooo`3oool0000001L0oooo00<00000
0?ooo`3oool01P3oool010000000oooo0?ooo`0000060?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`0000001P3oool00`000000oooo0?ooo`040?ooo`070000003oool000000?ooo`000000
oooo000000020?ooo`80000000<0oooo0000000000000`3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo0080oooo0P0000040?ooo`030000003oool0oooo00<0oooo0P0000050?ooo`03
0000003oool0oooo00l0oooo00<000000?ooo`3oool00P3oool003X0oooo00<000000?ooo`3oool0
1`3oool00`000000oooo0?ooo`020?ooo`040000003oool0oooo000001P0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo000000050?ooo`030000003oool000000080oooo00D000000?ooo`3o
ool0oooo000000060?ooo`030000003oool0oooo00<0oooo00<000000?ooo`0000000P3oool01000
0000oooo0000003oool2000000040?ooo`0000000000000000<0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0103o
ool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool00P3o
ool003X0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`020?ooo`030000003oool0
000001T0oooo00<000000?ooo`3oool01P3oool00`000000oooo000000050?ooo`030000003oool0
00000080oooo00D000000?ooo`3oool0oooo000000060?ooo`030000003oool0oooo00@0oooo0P00
00030?ooo`80000000<0oooo0000000000000P3oool2000000@0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00<0oooo0P0000040?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`020?ooo`00>P3oool0
0`000000oooo0?ooo`090?ooo`040000003oool0oooo0?ooo`8000006P3oool00`000000oooo0?oo
o`060?ooo`030000003oool0000000D0oooo00<000000?ooo`3oool00P0000040?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool0103oool2000000@0oooo00<000000?ooo`0000000`3o
ool2000000@0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`040?ooo`030000003o
ool0oooo00`0oooo00<000000?ooo`3oool00P3oool003T0oooo0P00000<0?ooo`8000007@3oool0
0`000000oooo0?ooo`060?ooo`030000003oool0000000D0oooo00<000000?ooo`3oool01P3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo0P0000040?ooo`040000003oool0
000000000080oooo00<000000?ooo`000000103oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`020?ooo`00B03oool0
0`000000oooo0?ooo`0K0?ooo`030000003oool0oooo00H0oooo0P0000060?ooo`030000003oool0
oooo00H0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo00<0oooo100000020?ooo`<00000103oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`8000001@3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo0080oooo00190?ooo`030000003oool0oooo
01X0oooo00<000000?ooo`3oool01P3oool2000000D0oooo00<000000?ooo`3oool01`3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool30000
0080oooo0`0000050?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`090?ooo`030000003oool0oooo0080oooo001:0?ooo`030000003oool0oooo01T0
oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`8000001@3oool300000080
oooo00@000000?ooo`00000000001@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`020?ooo`00B`3oool00`000000
oooo0?ooo`0H0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`070?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0103oool2000000D0
oooo100000000`3oool000000?ooo`02000000D0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool00P3oool004`0oooo
00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo0080oooo
00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo
0P0000050?ooo`040000003oool000000?ooo`8000000P3oool010000000oooo0000000000030?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo0080oooo001=0?ooo`030000003oool0000001H0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3o
ool0103oool00`000000oooo0?ooo`030?ooo`8000001@3oool300000080oooo00<000000?ooo`3o
ool00P0000020?ooo`040000003oool0oooo000000D0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo0P0000050?ooo`030000003oool0oooo
00H0oooo00<000000?ooo`3oool00P3oool004h0oooo00<000000?ooo`3oool05@3oool00`000000
oooo0?ooo`070?ooo`8000000`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool00P3oool00`000000oooo000000050?ooo`80000000<0oooo0000003o
ool00P0000000`3oool00000000000030?ooo`040000003oool0oooo000000D0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?oo
o`3oool00`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo0080oooo001?0?ooo`03
0000003oool0oooo01@0oooo00<000000?ooo`3oool01P3oool00`000000oooo000000020?ooo`03
0000003oool0oooo00X0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`020?ooo`04
0000003oool0oooo000000@0oooo00@000000?ooo`00000000000P3oool4000000<0oooo00@00000
0?ooo`3oool000001P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00D0oooo00<0
00000?ooo`3oool00P3oool00500oooo00<000000?ooo`3oool04`3oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo00<00000303oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo0080oooo00@000000?ooo`3oool00000103oool010000000oooo0000000000020?ooo`040000
003oool00000000000<0oooo00@000000?ooo`3oool000001`3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00<0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`3oool00@3oool00540oooo0P00000D0?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo0080oooo00@000000?ooo`3oool00000103oool010000000oooo000000000002
0?ooo`030000003oool00000008000000`3oool010000000oooo0?ooo`0000060?ooo`030000003o
ool0oooo00D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`010?ooo`00D`3oool0
0`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool03`3oool0
0`000000oooo0?ooo`050?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?oo
o`020?ooo`040000003oool0000000000080oooo0P0000000`3oool00000000000030?ooo`050000
003oool0oooo0?ooo`0000001P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool00`3oool2000000D0oooo00<000000?ooo`3oool00`3oool00`000000
oooo0?ooo`010?ooo`00E03oool200000140oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`0?0?ooo`030000003oool0oooo00D0oooo00D000000?ooo`3oool0oooo000000030?ooo`04
0000003oool0oooo000000<0oooo00<000000?ooo`3oool00P000000103oool000000?ooo`000004
0?ooo`@000001P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00<0oooo00<00000
0?ooo`3oool00@3oool005@0oooo0P00000A0?ooo`030000003oool0oooo00D0oooo00<000000?oo
o`3oool03`3oool00`000000oooo0?ooo`050?ooo`050000003oool0oooo0?ooo`0000000`3oool0
10000000oooo0?ooo`0000030?ooo`040000003oool0oooo00000080oooo00@000000?ooo`000000
00000P3oool01@000000oooo0?ooo`3oool0000000L0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo0040oooo001F0?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00D0
oooo00D000000?ooo`3oool0oooo000000030?ooo`040000003oool0oooo000000<0oooo0P000000
0`3oool000000?ooo`040?ooo`<000001@3oool00`000000oooo0?ooo`050?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool0103oool2000000D0oooo00<000000?ooo`3oool00`3oool0
1@000000oooo0?ooo`3oool0000000<0oooo001G0?ooo`030000003oool0oooo00d0oooo00<00000
0?ooo`3oool0103oool00`000000oooo0?ooo`0@0?ooo`030000003oool0oooo00D0oooo00D00000
0?ooo`3oool0oooo000000030?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`000000
1P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
1`3oool00`000000oooo0?ooo`010?ooo`00F03oool00`000000oooo0?ooo`0=0?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`050?ooo`050000003o
ool0oooo0?ooo`0000000`3oool010000000oooo0?ooo`0000040?ooo`<000001P3oool2000000L0
oooo0`0000040?ooo`030000003oool0oooo00@0oooo0P0000050?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`010?ooo`00F@3oool00`000000oooo
0?ooo`0<0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0403oool00`000000oooo
0?ooo`050?ooo`050000003oool0oooo0?ooo`000000103oool00`000000oooo000000040?ooo`<0
00001@3oool010000000oooo0000000000030?ooo`<0000000@0oooo0000000000000000103oool3
000000D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo0040oooo001J0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`0A0?ooo`030000003oool0oooo00@0oooo00@000000?ooo`3oool000001@3oool0
0`000000oooo000000040?ooo`<000001@3oool00`000000oooo0?ooo`04000000L0oooo10000003
0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`01
0?ooo`00F`3oool2000000/0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0A0?oo
o`030000003oool0oooo00D0oooo00<000000?ooo`0000001@3oool00`000000oooo000000040?oo
o`<00000103oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo00<000000?oo
o`3oool05@3oool00`000000oooo0?ooo`010?ooo`00G03oool00`000000oooo0?ooo`090?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`050?ooo`03
0000003oool0000000D0oooo0P0000050?ooo`030000003oool000000080000000<0oooo0000003o
ool01P3oool00`000000oooo0?ooo`0U0?ooo`030000003oool0oooo0040oooo001M0?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0B0?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`0000001@3oool2000000D0oooo00<000000?ooo`000000
0P3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo02@0oooo00<000000?ooo`3oool0
0@3oool005h0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`020?ooo`030000003o
ool0oooo0180oooo00<000000?ooo`3oool01@3oool00`000000oooo000000060?ooo`030000003o
ool0oooo00<0oooo00<000000?ooo`0000003@3oool00`000000oooo0?ooo`0S0?ooo`030000003o
ool0oooo0040oooo001O0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`0000001P3oool0
0`000000oooo0?ooo`030?ooo`<000003P3oool00`000000oooo0?ooo`0R0?ooo`030000003oool0
oooo0040oooo001P0?ooo`040000003oool0oooo000000D0oooo00@000000?ooo`3oool000005@3o
ool00`000000oooo0?ooo`050?ooo`030000003oool0000000H0oooo0P0000030?ooo`040000003o
ool00000000000l0oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`010?ooo`00F@3o
ool4000000@0oooo0P0000060?ooo`040000003oool0oooo000001D0oooo00<000000?ooo`3oool0
1@3oool00`000000oooo000000060?ooo`8000000`3oool010000000oooo00000000000@0?ooo`03
0000003oool0oooo0200oooo00<000000?ooo`3oool00@3oool005T0oooo00<000000?ooo`3oool0
1P3oool2000000D0oooo00@000000?ooo`3oool000005@3oool00`000000oooo0?ooo`060?ooo`80
00001@3oool00`000000oooo0?ooo`0300000080oooo0P00000A0?ooo`030000003oool0oooo0200
oooo0@0000010?ooo`40oooo001J0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0
0P3oool010000000oooo0?ooo`00000E0?ooo`030000003oool0oooo00H0oooo0P0000050?ooo`03
0000003oool0oooo00D0oooo0P00000B0?ooo`030000003oool0oooo01l0oooo0@0000010?ooo`40
oooo001K0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool00P3oool2000001H0oooo
00<000000?ooo`3oool01P3oool2000000D0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`0A0?ooo`030000003oool0oooo01h0oooo0@0000010?ooo`40oooo001I0?ooo`040000003o
ool0oooo000000T0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0D0?ooo`030000
003oool0oooo00H0oooo0P0000050?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0
4P3oool00`000000oooo0?ooo`0M0?ooo`4000000@3oool10?ooo`00FP3oool2000000/0oooo0P00
000J0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?oo
o`070?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool07@3oool100000040oooo0@3o
ool004L0oooo0P00000O0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool01P3oool0
0`000000oooo0?ooo`020?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool04`3oool0
0`000000oooo0?ooo`0L0?ooo`4000000@3oool10?ooo`00AP3oool010000000oooo0?ooo`00000O
0?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool01@3oool2000000@0oooo00<00000
0?ooo`3oool01`3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo01/0oooo0@000001
0?ooo`40oooo00160?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool05P3oool00`00
0000oooo0?ooo`050?ooo`800000103oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo
01D0oooo00<000000?ooo`3oool06P3oool100000040oooo0@3oool004D0oooo1@00000Q0?ooo`03
0000003oool0oooo01D0oooo00<000000?ooo`3oool01@3oool00`000000oooo000000020?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`0I0?ooo`40
00000@3oool10?ooo`00AP3oool00`000000oooo0?ooo`0S0?ooo`030000003oool0oooo01@0oooo
00<000000?ooo`3oool01@3oool00`000000oooo000000020?ooo`030000003oool0oooo00P0oooo
00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`0H0?ooo`4000000@3oool10?ooo`00K@3o
ool2000001@0oooo00<000000?ooo`3oool01@3oool00`000000oooo000000020?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool0603oool00`000000oooo0?ooo`0G0?ooo`4000000@3o
ool10?ooo`00KP3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo00D0oooo00<00000
0?ooo`3oool00P00000;0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool05P3oool1
00000040oooo0@3oool006l0oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`040?oo
o`030000003oool0oooo00d0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`0E0?oo
o`4000000@3oool10?ooo`00L03oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo01@0
oooo0@0000010?ooo`40oooo001a0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0
103oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo01`0oooo00<000000?ooo`3oool0
4`3oool100000040oooo0@3oool00780oooo0`00000?0?ooo`030000003oool0oooo00@0oooo00<0
00000?ooo`3oool03@3oool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo0180oooo0@00
00010?ooo`40oooo001c0?ooo`8000003`3oool00`000000oooo0?ooo`040?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`0B0?ooo`4000000@3oool0
07D0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo
00d0oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`0A0?ooo`4000000@3oool007H0
oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0
oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`0A0?ooo`4000000@3oool007L0oooo
00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo
00<000000?ooo`3oool0803oool00`000000oooo0?ooo`0@0?ooo`4000000@3oool007P0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00h0oooo00<0
00000?ooo`3oool08@3oool00`000000oooo0?ooo`0?0?ooo`4000000@3oool007T0oooo0P00000:
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0R
0?ooo`030000003oool0oooo00h0oooo0@0000010?ooo`00NP3oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo0080oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0S0?oo
o`030000003oool0oooo00d0oooo0@0000010?ooo`00N`3oool00`000000oooo0?ooo`070?ooo`03
0000003oool0oooo0080oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0T0?ooo`03
0000003oool0oooo00`0oooo0@0000010?ooo`00O03oool00`000000oooo0?ooo`060?ooo`030000
003oool0oooo0080oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0U0?ooo`030000
003oool0oooo00/0oooo0@0000010?ooo`00O@3oool00`000000oooo0?ooo`050?ooo`050000003o
ool0oooo0?ooo`0000004@3oool00`000000oooo0?ooo`0V0?ooo`030000003oool0oooo00X0oooo
0@0000010?ooo`00OP3oool3000000@0oooo00D000000?ooo`3oool0oooo0000000A0?ooo`030000
003oool0oooo02L0oooo00<000000?ooo`3oool02@3oool100000040oooo001g0?ooo`8000001P3o
ool00`000000oooo0?ooo`040?ooo`040000003oool0oooo00000140oooo00<000000?ooo`3oool0
:03oool00`000000oooo0?ooo`080?ooo`4000000@3oool007H0oooo00@000000?ooo`3oool00000
1P3oool00`000000oooo0?ooo`030?ooo`030000003oool000000180oooo00<000000?ooo`3oool0
:@3oool00`000000oooo0?ooo`070?ooo`4000000@3oool007P0oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`030?ooo`8000004P3oool00`000000oooo0?ooo`0Z0?ooo`030000003o
ool0oooo00H0oooo0@0000010?ooo`00N@3oool00`000000oooo0?ooo`060?ooo`030000003oool0
oooo0080oooo00<000000?ooo`3oool04P3oool00`000000oooo0?ooo`0Z0?ooo`030000003oool0
oooo00D0oooo0@0000010?ooo`00MP3oool010000000oooo0?ooo`0000090?ooo`<000005P3oool0
0`000000oooo0?ooo`0[0?ooo`030000003oool0oooo00@0oooo0@0000010?ooo`00M`3oool20000
00/0oooo0P00000F0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3oool0103oool10000
0000QP3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo02`0oooo00<000000?ooo`3o
ool0103oool100000000Q`3oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo02d0oooo
00<000000?ooo`3oool00`3oool100000000R03oool00`000000oooo0?ooo`0A0?ooo`030000003o
ool0oooo02h0oooo00<000000?ooo`3oool00P3oool100000000R@3oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo02l0oooo00@000000?ooo`3oool0oooo0@00000008X0oooo0P00000@
0?ooo`030000003oool0oooo0300oooo00<000000?ooo`3oool00@00000008/0oooo00<000000?oo
o`3oool03P3oool00`000000oooo0?ooo`0a0?ooo`4000000@3oool100000000S03oool00`000000
oooo0?ooo`0=0?ooo`030000003oool0oooo0380oooo0P00000008d0oooo00<000000?ooo`3oool0
303oool00`000000oooo0?ooo`0b0?ooo`800000002>0?ooo`030000003oool0oooo00/0oooo00<0
00000?ooo`3oool0;P3oool400000080oooo002?0?ooo`8000002`3oool00`000000oooo0?ooo`0Z
0?ooo`@000001P3oool00900oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0V0?oo
o`@000002P3oool00940oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0R0?ooo`@0
00003P3oool00980oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0N0?ooo`@00000
4P3oool009<0oooo0P0000070?ooo`030000003oool0oooo01X0oooo1000000F0?ooo`00U03oool2
000000H0oooo00<000000?ooo`3oool05`3oool3000001X0oooo002F0?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool04`3oool4000001d0oooo002G0?ooo`030000003oool0oooo0080
oooo00<000000?ooo`3oool03`3oool400000240oooo002H0?ooo`050000003oool0oooo0?ooo`00
00003@3oool4000002D0oooo002I0?ooo`040000003oool00000000000T0oooo1000000Y0?ooo`00
T`3oool3000000@0oooo00<000000?ooo`0000001@3oool4000002d0oooo002D0?ooo`030000003o
ool0oooo00@0oooo0P0000000`3oool000000000000200000340oooo002A0?ooo`D000001P3oool2
000003D0oooo002A0?ooo`040000003oool0oooo000003h0oooo002B0?ooo`030000003oool00000
03h0oooo002C0?ooo`800000?P3oool00=<0oooo003C0?ooo`00d`3oool00=<0oooo003C0?ooo`00
d`3oool00=<0oooo0000\
\>"],
  ImageRangeCache->{{{0, 210.75}, {286.938, 0}} -> {0.0366251, -0.0840194, \
0.00366745, 0.00366745}}]
}, Open  ]],

Cell["\<\
The plot above indicates why this technique is called the numerical \
\"method of lines\". \
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "The solution in between lines can be found by interpolation.",
  " ",
  "When ",
  StyleBox["NDSolve", "MR"],
  " computes the solution for the PDE, the result is a two-dimensional ",
  StyleBox["InterpolatingFunction.",
    FontFamily->"Courier"]
}], "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "This uses ",
  StyleBox["NDSolve", "MR"],
  " to compute the solution of the heat equation (1) directly."
}], "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(solution\  = \ 
      NDSolve[{D[u[x, t], t]\  \[Equal] \ \(1\/8\) D[u[x, t], x, x], 
          u[x, 0]\  \[Equal] \ 0, \ 
          u[0, t]\  \[Equal] \ 
            Sin[2\ \[Pi]\ t], \((D[u[x, t], x]\  /. 
                x \[Rule] 1)\)\  \[Equal] \ 0}, 
        u, {x, 0, 1}, {t, 0, 4}]\)], "Input",
  CellLabel->"In[9]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{"u", "\[Rule]", 
          TagBox[\(InterpolatingFunction[{{0.`, 1.`}, {0.`, 4.`}}, "<>"]\),
            False,
            Editable->False]}], "}"}], "}"}]], "Output",
  CellLabel->"Out[9]="]
}, Open  ]],

Cell["This creates a surface plot of the solution.", "MathCaption",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Plot3D[
        Evaluate[First[u[x, t]\  /. \ solution]], {x, 0, 1}, {t, 0, 4}, \ 
        PlotPoints \[Rule] {14, 36}, PlotRange \[Rule] All];\)\)], "Input",
  CellLabel->"In[10]:=",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-1.55041e-017 1.04977 -0.0679587 1.04977 [
[.02757 .24618 -6 -8.70276 ]
[.02757 .24618 0 .29724 ]
[.17646 .18758 -23.5978 -9 ]
[.17646 .18758 .40224 0 ]
[.3345 .12542 -16.7601 -9 ]
[.3345 .12542 1.2399 0 ]
[.50253 .05935 -21.0959 -9 ]
[.50253 .05935 2.90415 0 ]
[.68156 -0.01098 -4.96123 -9 ]
[.68156 -0.01098 1.03877 0 ]
[.70096 -0.00478 0 -6.26206 ]
[.70096 -0.00478 6 2.73794 ]
[.78313 .12104 0 -6.13858 ]
[.78313 .12104 6 2.86142 ]
[.8565 .23339 0 -6.03127 ]
[.8565 .23339 6 2.96873 ]
[.9224 .33431 0 -5.93715 ]
[.9224 .33431 6 3.06285 ]
[.98191 .42546 0 -5.85393 ]
[.98191 .42546 6 3.14607 ]
[.02414 .26497 -12 -2.7422 ]
[.02414 .26497 0 6.2578 ]
[.0157 .31787 -24 -2.78692 ]
[.0157 .31787 0 6.21308 ]
[.00696 .37261 -6 -2.83337 ]
[.00696 .37261 0 6.16663 ]
[-0.00209 .42931 -18 -2.88164 ]
[-0.00209 .42931 0 6.11836 ]
[-0.01146 .48806 -6 -2.93185 ]
[-0.01146 .48806 0 6.06815 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.18558 .19702 m
.19014 .20174 L
s
[(0.25)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(0.5)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(0.75)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(1)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.1552 .20892 m
.15797 .21172 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.27893 .16047 m
.28157 .16339 L
s
.3108 .14799 m
.31341 .15094 L
s
.37572 .12257 m
.37826 .12558 L
s
.40878 .10962 m
.41128 .11266 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.54521 .0562 m
.54755 .05937 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.68874 0 m
.68263 .00239 L
s
[(0)] .70096 -0.00478 -1 .39157 Mshowa
.7708 .12553 m
.76464 .12778 L
s
[(1)] .78313 .12104 -1 .36413 Mshowa
.84407 .23761 m
.83786 .23973 L
s
[(2)] .8565 .23339 -1 .34028 Mshowa
.9099 .3383 m
.90365 .34029 L
s
[(3)] .9224 .33431 -1 .31937 Mshowa
.96935 .42924 m
.96306 .43113 L
s
[(4)] .98191 .42546 -1 .30087 Mshowa
.125 Mabswid
.70593 .0263 m
.70226 .02771 L
s
.72272 .05198 m
.71904 .05338 L
s
.73912 .07706 m
.73543 .07845 L
s
.75514 .10158 m
.75145 .10294 L
s
.78611 .14895 m
.78241 .15028 L
s
.80108 .17185 m
.79737 .17317 L
s
.81573 .19425 m
.81201 .19555 L
s
.83006 .21617 m
.82633 .21745 L
s
.8578 .2586 m
.85407 .25986 L
s
.87123 .27915 m
.8675 .28039 L
s
.88439 .29928 m
.88065 .3005 L
s
.89727 .31899 m
.89353 .3202 L
s
.92227 .35722 m
.91851 .3584 L
s
.93439 .37576 m
.93063 .37693 L
s
.94627 .39394 m
.94251 .3951 L
s
.95792 .41176 m
.95416 .41291 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.03636 .26019 m
.04247 .25781 L
s
[(-1)] .02414 .26497 1 -0.39062 Mshowa
.02796 .3132 m
.03409 .31086 L
s
[(-0.5)] .0157 .31787 1 -0.38068 Mshowa
.01927 .36806 m
.02542 .36578 L
s
[(0)] .00696 .37261 1 -0.37036 Mshowa
.01026 .42487 m
.01644 .42265 L
s
[(0.5)] -0.00209 .42931 1 -0.35963 Mshowa
.00093 .48374 m
.00713 .48158 L
s
[(1)] -0.01146 .48806 1 -0.34848 Mshowa
.125 Mabswid
.0347 .27065 m
.03837 .26922 L
s
.03303 .28118 m
.03671 .27976 L
s
.03135 .29178 m
.03503 .29036 L
s
.02966 .30245 m
.03334 .30104 L
s
.02624 .32402 m
.02993 .32262 L
s
.02452 .33491 m
.0282 .33353 L
s
.02278 .34588 m
.02647 .3445 L
s
.02103 .35693 m
.02472 .35556 L
s
.01749 .37926 m
.02118 .3779 L
s
.0157 .39054 m
.0194 .38919 L
s
.0139 .4019 m
.0176 .40056 L
s
.01209 .41334 m
.01579 .41201 L
s
.00842 .43647 m
.01213 .43515 L
s
.00657 .44816 m
.01028 .44685 L
s
.00471 .45994 m
.00842 .45863 L
s
.00283 .4718 m
.00654 .47049 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
.59 .425 .627 r
.39515 .62854 .40169 .70185 .44302 .66479 .43514 .62188 Metetra
.574 .447 .67 r
.43514 .62188 .44302 .66479 .48416 .64658 .47573 .62458 Metetra
.555 .515 .765 r
.47573 .62458 .48416 .64658 .52576 .63828 .51728 .6288 Metetra
.505 .664 .933 r
.51728 .6288 .52576 .63828 .56815 .63385 .55986 .63047 Metetra
.371 .777 .983 r
.55986 .63047 .56815 .63385 .61139 .62965 .60336 .6281 Metetra
.348 .806 .937 r
.60336 .6281 .61139 .62965 .65547 .6239 .64766 .62175 Metetra
.466 .857 .976 r
.64766 .62175 .65547 .6239 .70031 .61599 .69269 .6122 Metetra
.577 .864 .995 r
.69269 .6122 .70031 .61599 .74588 .60606 .73839 .60041 Metetra
.633 .832 .977 r
.73839 .60041 .74588 .60606 .79214 .59453 .78478 .58725 Metetra
.653 .796 .951 r
.78478 .58725 .79214 .59453 .83912 .58188 .83189 .57336 Metetra
.659 .769 .931 r
.83189 .57336 .83912 .58188 .88683 .56854 .87975 .55917 Metetra
.659 .752 .919 r
.87975 .55917 .88683 .56854 .93532 .55483 .9284 .54491 Metetra
.658 .743 .913 r
.9284 .54491 .93532 .55483 .98461 .54091 .9779 .53071 Metetra
.51 .408 .674 r
.38754 .58813 .39515 .62854 .43514 .62188 .4265 .60862 Metetra
.345 .501 .87 r
.4265 .60862 .43514 .62188 .47573 .62458 .4669 .62522 Metetra
.341 0 0 r
.4669 .62522 .47573 .62458 .51728 .6288 .50865 .63427 Metetra
.374 0 0 r
.50865 .63427 .51728 .6288 .55986 .63047 .55152 .63543 Metetra
.216 0 0 r
.55152 .63543 .55986 .63047 .60336 .6281 .59527 .63008 Metetra
.59527 .63008 .60336 .6281 .64766 .62175 .63976 .62015 Metetra
.504 .886 .966 r
.63976 .62015 .64766 .62175 .69269 .6122 .68491 .60743 Metetra
.634 .856 .984 r
.68491 .60743 .69269 .6122 .73839 .60041 .73073 .59327 Metetra
.662 .801 .949 r
.73073 .59327 .73839 .60041 .78478 .58725 .77724 .57854 Metetra
.666 .766 .924 r
.77724 .57854 .78478 .58725 .83189 .57336 .82448 .56372 Metetra
.664 .747 .912 r
.82448 .56372 .83189 .57336 .87975 .55917 .87251 .549 Metetra
.661 .738 .906 r
.87251 .549 .87975 .55917 .9284 .54491 .92135 .53444 Metetra
.658 .733 .905 r
.92135 .53444 .9284 .54491 .9779 .53071 .97106 .52004 Metetra
.706 .284 .243 r
.37798 .59496 .38754 .58813 .4265 .60862 .41678 .6275 Metetra
.606 .256 .35 r
.41678 .6275 .4265 .60862 .4669 .62522 .45756 .64452 Metetra
.539 .185 .315 r
.45756 .64452 .4669 .62522 .50865 .63427 .49979 .64819 Metetra
.436 .035 .179 r
.49979 .64819 .50865 .63427 .55152 .63543 .54303 .64225 Metetra
.161 0 0 r
.54303 .64225 .55152 .63543 .59527 .63008 .58702 .63042 Metetra
.435 .873 .886 r
.58702 .63042 .59527 .63008 .63976 .62015 .63165 .61559 Metetra
.657 .866 .979 r
.63165 .61559 .63976 .62015 .68491 .60743 .67691 .59968 Metetra
.677 .788 .933 r
.67691 .59968 .68491 .60743 .73073 .59327 .72284 .58375 Metetra
.673 .748 .907 r
.72284 .58375 .73073 .59327 .77724 .57854 .76949 .56821 Metetra
.668 .732 .898 r
.76949 .56821 .77724 .57854 .82448 .56372 .81692 .55314 Metetra
.663 .729 .898 r
.81692 .55314 .82448 .56372 .87251 .549 .86516 .53842 Metetra
.66 .731 .902 r
.86516 .53842 .87251 .549 .92135 .53444 .91423 .52387 Metetra
.658 .734 .905 r
.91423 .52387 .92135 .53444 .97106 .52004 .96417 .50938 Metetra
.604 .328 .472 r
.36635 .64294 .37798 .59496 .41678 .6275 .40614 .66638 Metetra
.555 .267 .432 r
.40614 .66638 .41678 .6275 .45756 .64452 .44787 .66944 Metetra
.478 .15 .331 r
.44787 .66944 .45756 .64452 .49979 .64819 .49075 .65969 Metetra
.251 0 .004 r
.49075 .65969 .49979 .64819 .54303 .64225 .53436 .64334 Metetra
.496 .906 .874 r
.53436 .64334 .54303 .64225 .58702 .63042 .57854 .62463 Metetra
.701 .839 .949 r
.57854 .62463 .58702 .63042 .63165 .61559 .62328 .60595 Metetra
.691 .747 .894 r
.62328 .60595 .63165 .61559 .67691 .59968 .66866 .58836 Metetra
.677 .712 .875 r
.66866 .58836 .67691 .59968 .72284 .58375 .71474 .57206 Metetra
.668 .705 .875 r
.71474 .57206 .72284 .58375 .76949 .56821 .7616 .55682 Metetra
.662 .711 .884 r
.7616 .55682 .76949 .56821 .81692 .55314 .80926 .54225 Metetra
.659 .723 .896 r
.80926 .54225 .81692 .55314 .86516 .53842 .85775 .52797 Metetra
.657 .735 .907 r
.85775 .52797 .86516 .53842 .91423 .52387 .90709 .51367 Metetra
.656 .743 .914 r
.90709 .51367 .91423 .52387 .96417 .50938 .95727 .49915 Metetra
.546 .287 .474 r
.35354 .70682 .36635 .64294 .40614 .66638 .39525 .7026 Metetra
.468 .176 .385 r
.39525 .7026 .40614 .66638 .44787 .66944 .43817 .6834 Metetra
.176 0 .013 r
.43817 .6834 .44787 .66944 .49075 .65969 .48166 .65863 Metetra
.723 .954 .949 r
.48166 .65863 .49075 .65969 .53436 .64334 .52554 .63373 Metetra
.725 .752 .873 r
.52554 .63373 .53436 .64334 .57854 .62463 .56985 .61118 Metetra
.692 .68 .834 r
.56985 .61118 .57854 .62463 .62328 .60595 .61471 .5916 Metetra
.674 .663 .831 r
.61471 .5916 .62328 .60595 .66866 .58836 .66024 .57461 Metetra
.664 .669 .845 r
.66024 .57461 .66866 .58836 .71474 .57206 .70653 .55947 Metetra
.658 .688 .867 r
.70653 .55947 .71474 .57206 .7616 .55682 .75363 .5454 Metetra
.654 .71 .889 r
.75363 .5454 .7616 .55682 .80926 .54225 .80156 .53177 Metetra
.653 .731 .907 r
.80156 .53177 .80926 .54225 .85775 .52797 .85033 .51808 Metetra
.652 .748 .92 r
.85033 .51808 .85775 .52797 .90709 .51367 .89994 .50405 Metetra
.654 .758 .926 r
.89994 .50405 .90709 .51367 .95727 .49915 .95038 .48951 Metetra
.41 .135 .382 r
.34125 .75069 .35354 .70682 .39525 .7026 .38501 .71318 Metetra
.38501 .71318 .39525 .7026 .43817 .6834 .42886 .6747 Metetra
.785 .772 .842 r
.42886 .6747 .43817 .6834 .48166 .65863 .47268 .64107 Metetra
.709 .638 .776 r
.47268 .64107 .48166 .65863 .52554 .63373 .51666 .61384 Metetra
.677 .605 .768 r
.51666 .61384 .52554 .63373 .56985 .61118 .56103 .59238 Metetra
.661 .608 .784 r
.56103 .59238 .56985 .61118 .61471 .5916 .60602 .5752 Metetra
.654 .63 .814 r
.60602 .5752 .61471 .5916 .66024 .57461 .65174 .56073 Metetra
.65 .663 .85 r
.65174 .56073 .66024 .57461 .70653 .55947 .69828 .54764 Metetra
.647 .699 .883 r
.69828 .54764 .70653 .55947 .75363 .5454 .74565 .53498 Metetra
.646 .729 .909 r
.74565 .53498 .75363 .5454 .80156 .53177 .79386 .52216 Metetra
.646 .75 .925 r
.79386 .52216 .80156 .53177 .85033 .51808 .8429 .50885 Metetra
.648 .764 .933 r
.8429 .50885 .85033 .51808 .89994 .50405 .89277 .49491 Metetra
.651 .771 .937 r
.89277 .49491 .89994 .50405 .95038 .48951 .94347 .48028 Metetra
.877 .821 .79 r
.33118 .74605 .34125 .75069 .38501 .71318 .37609 .68772 Metetra
.724 .595 .712 r
.37609 .68772 .38501 .71318 .42886 .6747 .42014 .64331 Metetra
.675 .545 .699 r
.42014 .64331 .42886 .6747 .47268 .64107 .46386 .61152 Metetra
.653 .539 .712 r
.46386 .61152 .47268 .64107 .51666 .61384 .50775 .58923 Metetra
.643 .557 .743 r
.50775 .58923 .51666 .61384 .56103 .59238 .55216 .57312 Metetra
.639 .594 .788 r
.55216 .57312 .56103 .59238 .60602 .5752 .5973 .5604 Metetra
.637 .644 .84 r
.5973 .5604 .60602 .5752 .65174 .56073 .64325 .54904 Metetra
.637 .696 .888 r
.64325 .54904 .65174 .56073 .69828 .54764 .69005 .5378 Metetra
.636 .737 .921 r
.69005 .5378 .69828 .54764 .74565 .53498 .73768 .52601 Metetra
.638 .761 .937 r
.73768 .52601 .74565 .53498 .79386 .52216 .78614 .51342 Metetra
.642 .772 .942 r
.78614 .51342 .79386 .52216 .8429 .50885 .83543 .5 Metetra
.647 .775 .942 r
.83543 .5 .8429 .50885 .89277 .49491 .88554 .48584 Metetra
.651 .775 .94 r
.88554 .48584 .89277 .49491 .94347 .48028 .93648 .47099 Metetra
.673 .5 .643 r
.32389 .69052 .33118 .74605 .37609 .68772 .36838 .63519 Metetra
.643 .481 .649 r
.36838 .63519 .37609 .68772 .42014 .64331 .4118 .60092 Metetra
.627 .486 .671 r
.4118 .60092 .42014 .64331 .46386 .61152 .45509 .58041 Metetra
.619 .512 .711 r
.45509 .58041 .46386 .61152 .50775 .58923 .49881 .5677 Metetra
.617 .564 .772 r
.49881 .5677 .50775 .58923 .55216 .57312 .54328 .55846 Metetra
.618 .637 .847 r
.54328 .55846 .55216 .57312 .5973 .5604 .5886 .54997 Metetra
.617 .713 .914 r
.5886 .54997 .5973 .5604 .64325 .54904 .6348 .54073 Metetra
.618 .765 .95 r
.6348 .54073 .64325 .54904 .69005 .5378 .68184 .53016 Metetra
.624 .788 .961 r
.68184 .53016 .69005 .5378 .73768 .52601 .7297 .51819 Metetra
.635 .791 .957 r
.7297 .51819 .73768 .52601 .78614 .51342 .77837 .50504 Metetra
.644 .785 .949 r
.77837 .50504 .78614 .51342 .83543 .5 .82785 .491 Metetra
.65 .775 .94 r
.82785 .491 .83543 .5 .88554 .48584 .87817 .47632 Metetra
.653 .768 .934 r
.87817 .47632 .88554 .48584 .93648 .47099 .92936 .46118 Metetra
.614 .436 .618 r
.31829 .60965 .32389 .69052 .36838 .63519 .36103 .5782 Metetra
.599 .444 .643 r
.36103 .5782 .36838 .63519 .4118 .60092 .40334 .56429 Metetra
.59 .475 .69 r
.40334 .56429 .4118 .60092 .45509 .58041 .44611 .55865 Metetra
.585 .544 .775 r
.44611 .55865 .45509 .58041 .49881 .5677 .48974 .55535 Metetra
.579 .657 .89 r
.48974 .55535 .49881 .5677 .54328 .55846 .53434 .55112 Metetra
.561 .766 .974 r
.53434 .55112 .54328 .55846 .5886 .54997 .57989 .54452 Metetra
.561 .821 .994 r
.57989 .54452 .5886 .54997 .6348 .54073 .62633 .53529 Metetra
.59 .835 .991 r
.62633 .53529 .6348 .54073 .68184 .53016 .67358 .52379 Metetra
.623 .824 .978 r
.67358 .52379 .68184 .53016 .7297 .51819 .72163 .5106 Metetra
.643 .803 .96 r
.72163 .5106 .7297 .51819 .77837 .50504 .77048 .4963 Metetra
.653 .781 .942 r
.77048 .4963 .77837 .50504 .82785 .491 .82014 .48134 Metetra
.656 .765 .93 r
.82014 .48134 .82785 .491 .87817 .47632 .87065 .46602 Metetra
.656 .754 .922 r
.87065 .46602 .87817 .47632 .92936 .46118 .92206 .45054 Metetra
.566 .406 .624 r
.3123 .53995 .31829 .60965 .36103 .5782 .35289 .54019 Metetra
.545 .44 .686 r
.35289 .54019 .36103 .5782 .40334 .56429 .3942 .54674 Metetra
.51 .544 .827 r
.3942 .54674 .40334 .56429 .44611 .55865 .4367 .5524 Metetra
.385 .717 .988 r
.4367 .5524 .44611 .55865 .48974 .55535 .48044 .55384 Metetra
.241 .744 .895 r
.48044 .55384 .48974 .55535 .53434 .55112 .52529 .55024 Metetra
.322 .802 .901 r
.52529 .55024 .53434 .55112 .57989 .54452 .57111 .54217 Metetra
.489 .869 .977 r
.57111 .54217 .57989 .54452 .62633 .53529 .61777 .53073 Metetra
.6 .864 .993 r
.61777 .53073 .62633 .53529 .67358 .52379 .66521 .51708 Metetra
.644 .825 .969 r
.66521 .51708 .67358 .52379 .72163 .5106 .71342 .50214 Metetra
.658 .789 .944 r
.71342 .50214 .72163 .5106 .77048 .4963 .76242 .48658 Metetra
.661 .764 .926 r
.76242 .48658 .77048 .4963 .82014 .48134 .81226 .47077 Metetra
.66 .749 .916 r
.81226 .47077 .82014 .48134 .87065 .46602 .86297 .45492 Metetra
.658 .741 .911 r
.86297 .45492 .87065 .46602 .92206 .45054 .9146 .43914 Metetra
.423 .387 .714 r
.30397 .50953 .3123 .53995 .35289 .54019 .34305 .53461 Metetra
0 .362 .748 r
.34305 .53461 .35289 .54019 .3942 .54674 .38402 .55237 Metetra
.462 0 0 r
.38402 .55237 .3942 .54674 .4367 .5524 .42674 .56043 Metetra
.371 0 0 r
.42674 .56043 .4367 .5524 .48044 .55384 .47087 .55951 Metetra
.154 0 0 r
.47087 .55951 .48044 .55384 .52529 .55024 .51609 .55173 Metetra
.261 .769 .772 r
.51609 .55173 .52529 .55024 .57111 .54217 .56218 .53942 Metetra
.571 .895 .985 r
.56218 .53942 .57111 .54217 .61777 .53073 .60904 .52456 Metetra
.654 .84 .972 r
.60904 .52456 .61777 .53073 .66521 .51708 .65664 .5085 Metetra
.668 .787 .938 r
.65664 .5085 .66521 .51708 .71342 .50214 .70501 .49208 Metetra
.667 .757 .918 r
.70501 .49208 .71342 .50214 .76242 .48658 .75419 .47568 Metetra
.664 .742 .909 r
.75419 .47568 .76242 .48658 .81226 .47077 .80423 .45943 Metetra
.661 .736 .906 r
.80423 .45943 .81226 .47077 .86297 .45492 .85516 .44334 Metetra
.658 .734 .906 r
.85516 .44334 .86297 .45492 .9146 .43914 .90702 .42736 Metetra
.658 .298 .351 r
.29228 .52788 .30397 .50953 .34305 .53461 .3313 .5602 Metetra
.577 .237 .357 r
.3313 .5602 .34305 .53461 .38402 .55237 .37291 .57452 Metetra
.502 .142 .288 r
.37291 .57452 .38402 .55237 .42674 .56043 .41639 .5746 Metetra
.362 0 .095 r
.41639 .5746 .42674 .56043 .47087 .55951 .46111 .56511 Metetra
.46111 .56511 .47087 .55951 .51609 .55173 .50673 .55021 Metetra
.577 .918 .966 r
.50673 .55021 .51609 .55173 .56218 .53942 .55308 .5329 Metetra
.678 .835 .959 r
.55308 .5329 .56218 .53942 .60904 .52456 .60012 .51499 Metetra
.68 .767 .917 r
.60012 .51499 .60904 .52456 .65664 .5085 .64788 .4974 Metetra
.672 .737 .899 r
.64788 .4974 .65664 .5085 .70501 .49208 .69644 .4804 Metetra
.666 .727 .895 r
.69644 .4804 .70501 .49208 .75419 .47568 .74582 .46394 Metetra
.662 .727 .898 r
.74582 .46394 .75419 .47568 .80423 .45943 .79609 .4478 Metetra
.659 .732 .904 r
.79609 .4478 .80423 .45943 .85516 .44334 .84727 .43177 Metetra
.657 .736 .908 r
.84727 .43177 .85516 .44334 .90702 .42736 .89938 .41569 Metetra
.575 .294 .453 r
.2776 .58359 .29228 .52788 .3313 .5602 .31829 .6012 Metetra
.518 .217 .396 r
.31829 .6012 .3313 .5602 .37291 .57452 .36143 .59823 Metetra
.405 .05 .247 r
.36143 .59823 .37291 .57452 .41639 .5746 .40596 .58336 Metetra
.40596 .58336 .41639 .5746 .46111 .56511 .45133 .56313 Metetra
.67 .934 .968 r
.45133 .56313 .46111 .56511 .50673 .55021 .49729 .54164 Metetra
.707 .791 .916 r
.49729 .54164 .50673 .55021 .55308 .5329 .54383 .52098 Metetra
.687 .722 .876 r
.54383 .52098 .55308 .5329 .60012 .51499 .59102 .50189 Metetra
.673 .7 .867 r
.59102 .50189 .60012 .51499 .64788 .4974 .63896 .48432 Metetra
.664 .7 .873 r
.63896 .48432 .64788 .4974 .69644 .4804 .68772 .46787 Metetra
.659 .711 .886 r
.68772 .46787 .69644 .4804 .74582 .46394 .73736 .45205 Metetra
.656 .725 .9 r
.73736 .45205 .74582 .46394 .79609 .4478 .7879 .43642 Metetra
.655 .738 .911 r
.7879 .43642 .79609 .4478 .84727 .43177 .83935 .42065 Metetra
.655 .747 .918 r
.83935 .42065 .84727 .43177 .89938 .41569 .89172 .40452 Metetra
.503 .232 .435 r
.26179 .64698 .2776 .58359 .31829 .6012 .30542 .63314 Metetra
.374 .053 .287 r
.30542 .63314 .31829 .6012 .36143 .59823 .35038 .60687 Metetra
.35038 .60687 .36143 .59823 .40596 .58336 .39587 .57745 Metetra
.765 .851 .916 r
.39587 .57745 .40596 .58336 .45133 .56313 .44166 .5497 Metetra
.713 .702 .839 r
.44166 .5497 .45133 .56313 .49729 .54164 .4878 .52544 Metetra
.682 .657 .819 r
.4878 .52544 .49729 .54164 .54383 .52098 .53446 .50472 Metetra
.667 .652 .826 r
.53446 .50472 .54383 .52098 .59102 .50189 .5818 .4868 Metetra
.658 .666 .847 r
.5818 .4868 .59102 .50189 .63896 .48432 .62994 .47072 Metetra
.653 .69 .871 r
.62994 .47072 .63896 .48432 .68772 .46787 .67894 .45559 Metetra
.651 .715 .895 r
.67894 .45559 .68772 .46787 .73736 .45205 .72884 .44073 Metetra
.65 .736 .912 r
.72884 .44073 .73736 .45205 .7879 .43642 .77966 .42566 Metetra
.651 .752 .924 r
.77966 .42566 .7879 .43642 .83935 .42065 .8314 .41012 Metetra
.653 .762 .929 r
.8314 .41012 .83935 .42065 .89172 .40452 .88405 .39393 Metetra
.226 0 .217 r
.24771 .68096 .26179 .64698 .30542 .63314 .29421 .63438 Metetra
.853 .971 .87 r
.29421 .63438 .30542 .63314 .35038 .60687 .34043 .59095 Metetra
.747 .679 .785 r
.34043 .59095 .35038 .60687 .39587 .57745 .38632 .55495 Metetra
.69 .602 .753 r
.38632 .55495 .39587 .57745 .44166 .5497 .43217 .52673 Metetra
.665 .587 .759 r
.43217 .52673 .44166 .5497 .4878 .52544 .47831 .50483 Metetra
.652 .601 .784 r
.47831 .50483 .4878 .52544 .53446 .50472 .52503 .48727 Metetra
.647 .631 .82 r
.52503 .48727 .53446 .50472 .5818 .4868 .57251 .47222 Metetra
.644 .67 .86 r
.57251 .47222 .5818 .4868 .62994 .47072 .62086 .45827 Metetra
.643 .707 .894 r
.62086 .45827 .62994 .47072 .67894 .45559 .67012 .44449 Metetra
.643 .737 .917 r
.67012 .44449 .67894 .45559 .72884 .44073 .72029 .43033 Metetra
.645 .756 .93 r
.72029 .43033 .72884 .44073 .77966 .42566 .77138 .41553 Metetra
.648 .767 .936 r
.77138 .41553 .77966 .42566 .8314 .41012 .82339 .39997 Metetra
.651 .773 .938 r
.82339 .39997 .8314 .41012 .88405 .39393 .87632 .38364 Metetra
.789 .662 .724 r
.23763 .66145 .24771 .68096 .29421 .63438 .28537 .59904 Metetra
.694 .549 .685 r
.28537 .59904 .29421 .63438 .34043 .59095 .33162 .55377 Metetra
.657 .52 .685 r
.33162 .55377 .34043 .59095 .38632 .55495 .37717 .52245 Metetra
.639 .524 .707 r
.37717 .52245 .38632 .55495 .43217 .52673 .42272 .50086 Metetra
.631 .552 .747 r
.42272 .50086 .43217 .52673 .47831 .50483 .46874 .48514 Metetra
.63 .599 .8 r
.46874 .48514 .47831 .50483 .52503 .48727 .51551 .4723 Metetra
.63 .658 .858 r
.51551 .4723 .52503 .48727 .57251 .47222 .56317 .46032 Metetra
.631 .712 .905 r
.56317 .46032 .57251 .47222 .62086 .45827 .61175 .44802 Metetra
.633 .75 .932 r
.61175 .44802 .62086 .45827 .67012 .44449 .66126 .43487 Metetra
.637 .769 .943 r
.66126 .43487 .67012 .44449 .72029 .43033 .71168 .42072 Metetra
.643 .776 .944 r
.71168 .42072 .72029 .43033 .77138 .41553 .76302 .40564 Metetra
.648 .776 .941 r
.76302 .40564 .77138 .41553 .82339 .39997 .81529 .38974 Metetra
.652 .774 .938 r
.81529 .38974 .82339 .39997 .87632 .38364 .86849 .37314 Metetra
.65 .47 .627 r
.2316 .59396 .23763 .66145 .28537 .59904 .27824 .54089 Metetra
.625 .46 .639 r
.27824 .54089 .28537 .59904 .33162 .55377 .32328 .5095 Metetra
.611 .472 .668 r
.32328 .5095 .33162 .55377 .37717 .52245 .36798 .49134 Metetra
.605 .509 .719 r
.36798 .49134 .37717 .52245 .42272 .50086 .41309 .48 Metetra
.605 .575 .794 r
.41309 .48 .42272 .50086 .46874 .48514 .45901 .47115 Metetra
.608 .662 .877 r
.45901 .47115 .46874 .48514 .51551 .4723 .50589 .4622 Metetra
.608 .739 .938 r
.50589 .4622 .51551 .4723 .56317 .46032 .55375 .4519 Metetra
.613 .784 .963 r
.55375 .4519 .56317 .46032 .61175 .44802 .60258 .43987 Metetra
.625 .798 .965 r
.60258 .43987 .61175 .44802 .66126 .43487 .65232 .42623 Metetra
.638 .795 .958 r
.65232 .42623 .66126 .43487 .71168 .42072 .70298 .41129 Metetra
.647 .784 .947 r
.70298 .41129 .71168 .42072 .76302 .40564 .75454 .39543 Metetra
.652 .773 .937 r
.75454 .39543 .76302 .40564 .81529 .38974 .80704 .37893 Metetra
.654 .765 .931 r
.80704 .37893 .81529 .38974 .86849 .37314 .86049 .36197 Metetra
.593 .414 .608 r
.22732 .50977 .2316 .59396 .27824 .54089 .27116 .48469 Metetra
.578 .428 .641 r
.27116 .48469 .27824 .54089 .32328 .5095 .31442 .4752 Metetra
.568 .472 .705 r
.31442 .4752 .32328 .5095 .36798 .49134 .35821 .472 Metetra
.563 .568 .816 r
.35821 .472 .36798 .49134 .41309 .48 .40302 .46949 Metetra
.549 .704 .942 r
.40302 .46949 .41309 .48 .45901 .47115 .449 .4648 Metetra
.532 .801 .995 r
.449 .4648 .45901 .47115 .50589 .4622 .4961 .45695 Metetra
.552 .839 .998 r
.4961 .45695 .50589 .4622 .55375 .4519 .54423 .44603 Metetra
.596 .843 .991 r
.54423 .44603 .55375 .4519 .60258 .43987 .5933 .43264 Metetra
.631 .825 .975 r
.5933 .43264 .60258 .43987 .65232 .42623 .64327 .41751 Metetra
.649 .8 .955 r
.64327 .41751 .65232 .42623 .70298 .41129 .69413 .40128 Metetra
.656 .777 .938 r
.69413 .40128 .70298 .41129 .75454 .39543 .7459 .38443 Metetra
.657 .761 .926 r
.7459 .38443 .75454 .39543 .80704 .37893 .79862 .36725 Metetra
.656 .751 .92 r
.79862 .36725 .80704 .37893 .86049 .36197 .85232 .34992 Metetra
.533 .384 .624 r
.22156 .44574 .22732 .50977 .27116 .48469 .26233 .45264 Metetra
.501 .437 .717 r
.26233 .45264 .27116 .48469 .31442 .4752 .30413 .46251 Metetra
.417 .595 .918 r
.30413 .46251 .31442 .4752 .35821 .472 .34749 .469 Metetra
.186 .691 .906 r
.34749 .469 .35821 .472 .40302 .46949 .3924 .46962 Metetra
.158 .699 .792 r
.3924 .46962 .40302 .46949 .449 .4648 .43869 .46428 Metetra
.336 .814 .89 r
.43869 .46428 .449 .4648 .4961 .45695 .48613 .45407 Metetra
.526 .881 .983 r
.48613 .45407 .4961 .45695 .54423 .44603 .53456 .44041 Metetra
.622 .86 .988 r
.53456 .44041 .54423 .44603 .5933 .43264 .58388 .42461 Metetra
.654 .816 .961 r
.58388 .42461 .5933 .43264 .64327 .41751 .63406 .40765 Metetra
.662 .781 .937 r
.63406 .40765 .64327 .41751 .69413 .40128 .68512 .39015 Metetra
.662 .759 .922 r
.68512 .39015 .69413 .40128 .7459 .38443 .7371 .37247 Metetra
.66 .746 .914 r
.7371 .37247 .7459 .38443 .79862 .36725 .79003 .35478 Metetra
.657 .739 .91 r
.79003 .35478 .79862 .36725 .85232 .34992 .84398 .33714 Metetra
.168 .333 .787 r
.21176 .42632 .22156 .44574 .26233 .45264 .25065 .45496 Metetra
.533 0 0 r
.25065 .45496 .26233 .45264 .30413 .46251 .29211 .47288 Metetra
.465 0 .045 r
.29211 .47288 .30413 .46251 .34749 .469 .33584 .47912 Metetra
.331 0 0 r
.33584 .47912 .34749 .469 .3924 .46962 .38134 .4755 Metetra
.049 0 0 r
.38134 .4755 .3924 .46962 .43869 .46428 .42817 .46483 Metetra
.395 .852 .87 r
.42817 .46483 .43869 .46428 .48613 .45407 .47602 .44982 Metetra
.621 .889 .988 r
.47602 .44982 .48613 .45407 .53456 .44041 .52474 .43255 Metetra
.667 .822 .958 r
.52474 .43255 .53456 .44041 .58388 .42461 .57429 .41438 Metetra
.671 .775 .928 r
.57429 .41438 .58388 .42461 .63406 .40765 .62467 .39603 Metetra
.667 .75 .912 r
.62467 .39603 .63406 .40765 .68512 .39015 .67595 .37783 Metetra
.663 .739 .906 r
.67595 .37783 .68512 .39015 .7371 .37247 .72815 .3598 Metetra
.66 .735 .905 r
.72815 .3598 .7371 .37247 .79003 .35478 .78133 .3419 Metetra
.657 .734 .906 r
.78133 .3419 .79003 .35478 .84398 .33714 .83553 .32405 Metetra
.617 .277 .373 r
.197 .45584 .21176 .42632 .25065 .45496 .23621 .48642 Metetra
.541 .201 .34 r
.23621 .48642 .25065 .45496 .29211 .47288 .2788 .49695 Metetra
.45 .076 .237 r
.2788 .49695 .29211 .47288 .33584 .47912 .32371 .4927 Metetra
.248 0 0 r
.32371 .4927 .33584 .47912 .38134 .4755 .37014 .47924 Metetra
.291 .777 .704 r
.37014 .47924 .38134 .4755 .42817 .46483 .41758 .46102 Metetra
.656 .905 .981 r
.41758 .46102 .42817 .46483 .47602 .44982 .46582 .44105 Metetra
.688 .805 .937 r
.46582 .44105 .47602 .44982 .52474 .43255 .51479 .421 Metetra
.68 .75 .904 r
.51479 .421 .52474 .43255 .57429 .41438 .56454 .40157 Metetra
.67 .728 .893 r
.56454 .40157 .57429 .41438 .62467 .39603 .61514 .38289 Metetra
.664 .723 .893 r
.61514 .38289 .62467 .39603 .67595 .37783 .66663 .36478 Metetra
.66 .727 .899 r
.66663 .36478 .67595 .37783 .72815 .3598 .71909 .34695 Metetra
.657 .733 .906 r
.71909 .34695 .72815 .3598 .78133 .3419 .77253 .32914 Metetra
.656 .739 .911 r
.77253 .32914 .78133 .3419 .83553 .32405 .827 .31118 Metetra
.54 .25 .423 r
.17848 .5176 .197 .45584 .23621 .48642 .22042 .52794 Metetra
.464 .146 .341 r
.22042 .52794 .23621 .48642 .2788 .49695 .26533 .51819 Metetra
.281 0 .103 r
.26533 .51819 .2788 .49695 .32371 .4927 .31182 .49791 Metetra
.378 .823 .692 r
.31182 .49791 .32371 .4927 .37014 .47924 .35917 .47373 Metetra
.723 .877 .953 r
.35917 .47373 .37014 .47924 .41758 .46102 .40711 .44945 Metetra
.705 .751 .888 r
.40711 .44945 .41758 .46102 .46582 .44105 .4556 .42677 Metetra
.682 .703 .863 r
.4556 .42677 .46582 .44105 .51479 .421 .50475 .40606 Metetra
.668 .691 .863 r
.50475 .40606 .51479 .421 .56454 .40157 .55467 .38703 Metetra
.66 .698 .874 r
.55467 .38703 .56454 .40157 .61514 .38289 .60547 .36912 Metetra
.656 .712 .889 r
.60547 .36912 .61514 .38289 .66663 .36478 .65722 .35174 Metetra
.654 .728 .903 r
.65722 .35174 .66663 .36478 .71909 .34695 .70994 .33444 Metetra
.654 .741 .914 r
.70994 .33444 .71909 .34695 .77253 .32914 .76368 .31687 Metetra
.655 .751 .92 r
.76368 .31687 .77253 .32914 .827 .31118 .81843 .29883 Metetra
.439 .149 .374 r
.15943 .57787 .17848 .5176 .22042 .52794 .20558 .55381 Metetra
.189 0 .094 r
.20558 .55381 .22042 .52794 .26533 .51819 .25302 .52053 Metetra
.727 .987 .887 r
.25302 .52053 .26533 .51819 .31182 .49791 .30079 .48674 Metetra
.751 .763 .862 r
.30079 .48674 .31182 .49791 .35917 .47373 .34867 .45641 Metetra
.699 .666 .814 r
.34867 .45641 .35917 .47373 .40711 .44945 .39679 .43056 Metetra
.673 .641 .81 r
.39679 .43056 .40711 .44945 .4556 .42677 .44536 .40869 Metetra
.659 .646 .826 r
.44536 .40869 .4556 .42677 .50475 .40606 .49461 .38969 Metetra
.653 .667 .851 r
.49461 .38969 .50475 .40606 .55467 .38703 .5447 .37239 Metetra
.65 .694 .878 r
.5447 .37239 .55467 .38703 .60547 .36912 .59572 .35586 Metetra
.648 .72 .901 r
.59572 .35586 .60547 .36912 .65722 .35174 .64772 .33939 Metetra
.649 .741 .917 r
.64772 .33939 .65722 .35174 .70994 .33444 .70073 .32256 Metetra
.651 .755 .926 r
.70073 .32256 .70994 .33444 .76368 .31687 .75476 .30513 Metetra
.653 .764 .931 r
.75476 .30513 .76368 .31687 .81843 .29883 .80981 .28698 Metetra
.14398 .59966 .15943 .57787 .20558 .55381 .19376 .54475 Metetra
.825 .792 .82 r
.19376 .54475 .20558 .55381 .25302 .52053 .24266 .49721 Metetra
.717 .619 .748 r
.24266 .49721 .25302 .52053 .30079 .48674 .29076 .45951 Metetra
.673 .575 .737 r
.29076 .45951 .30079 .48674 .34867 .45641 .33856 .43068 Metetra
.652 .574 .754 r
.33856 .43068 .34867 .45641 .39679 .43056 .38652 .40844 Metetra
.643 .597 .788 r
.38652 .40844 .39679 .43056 .44536 .40869 .43504 .39038 Metetra
.64 .636 .831 r
.43504 .39038 .44536 .40869 .49461 .38969 .48437 .37447 Metetra
.64 .68 .873 r
.48437 .37447 .49461 .38969 .5447 .37239 .53463 .35928 Metetra
.64 .719 .905 r
.53463 .35928 .5447 .37239 .59572 .35586 .58588 .34394 Metetra
.642 .746 .924 r
.58588 .34394 .59572 .35586 .64772 .33939 .63815 .32797 Metetra
.645 .761 .933 r
.63815 .32797 .64772 .33939 .70073 .32256 .69143 .31121 Metetra
.649 .768 .936 r
.69143 .31121 .70073 .32256 .75476 .30513 .74575 .29364 Metetra
.653 .772 .936 r
.74575 .29364 .75476 .30513 .80981 .28698 .8011 .27525 Metetra
.734 .578 .68 r
.13465 .56471 .14398 .59966 .19376 .54475 .18544 .49992 Metetra
.668 .512 .663 r
.18544 .49992 .19376 .54475 .24266 .49721 .23392 .45496 Metetra
.638 .497 .674 r
.23392 .45496 .24266 .49721 .29076 .45951 .28128 .42474 Metetra
.624 .512 .706 r
.28128 .42474 .29076 .45951 .33856 .43068 .32846 .40409 Metetra
.619 .551 .756 r
.32846 .40409 .33856 .43068 .38652 .40844 .37609 .38869 Metetra
.621 .611 .819 r
.37609 .38869 .38652 .40844 .43504 .39038 .42453 .37544 Metetra
.624 .677 .88 r
.42453 .37544 .43504 .39038 .48437 .37447 .47396 .36238 Metetra
.628 .732 .922 r
.47396 .36238 .48437 .37447 .53463 .35928 .52443 .34851 Metetra
.633 .764 .942 r
.52443 .34851 .53463 .35928 .58588 .34394 .57594 .33345 Metetra
.64 .777 .946 r
.57594 .33345 .58588 .34394 .63815 .32797 .62847 .31719 Metetra
.646 .778 .944 r
.62847 .31719 .63815 .32797 .69143 .31121 .68202 .29992 Metetra
.651 .773 .938 r
.68202 .29992 .69143 .31121 .74575 .29364 .7366 .28185 Metetra
.654 .769 .934 r
.7366 .28185 .74575 .29364 .8011 .27525 .79224 .26312 Metetra
.626 .44 .611 r
.1305 .48673 .13465 .56471 .18544 .49992 .17908 .43759 Metetra
.603 .437 .629 r
.17908 .43759 .18544 .49992 .23392 .45496 .22553 .40995 Metetra
.591 .458 .668 r
.22553 .40995 .23392 .45496 .28128 .42474 .27147 .39446 Metetra
.588 .51 .735 r
.27147 .39446 .28128 .42474 .32846 .40409 .31788 .38445 Metetra
.592 .597 .826 r
.31788 .38445 .32846 .40409 .37609 .38869 .36524 .37567 Metetra
.596 .698 .914 r
.36524 .37567 .37609 .38869 .42453 .37544 .41373 .36578 Metetra
.6 .771 .961 r
.41373 .36578 .42453 .37544 .47396 .36238 .46336 .35382 Metetra
.613 .803 .973 r
.46336 .35382 .47396 .36238 .52443 .34851 .51409 .33968 Metetra
.63 .808 .968 r
.51409 .33968 .52443 .34851 .57594 .33345 .56587 .3237 Metetra
.645 .797 .956 r
.56587 .3237 .57594 .33345 .62847 .31719 .61866 .30635 Metetra
.653 .782 .943 r
.61866 .30635 .62847 .31719 .68202 .29992 .67245 .28809 Metetra
.655 .768 .932 r
.67245 .28809 .68202 .29992 .7366 .28185 .72729 .26928 Metetra
.656 .758 .925 r
.72729 .26928 .7366 .28185 .79224 .26312 .7832 .25012 Metetra
.566 .387 .598 r
.1276 .40159 .1305 .48673 .17908 .43759 .17203 .38377 Metetra
.55 .409 .643 r
.17203 .38377 .17908 .43759 .22553 .40995 .21589 .37901 Metetra
.539 .476 .733 r
.21589 .37901 .22553 .40995 .27147 .39446 .26051 .37818 Metetra
.529 .611 .878 r
.26051 .37818 .27147 .39446 .31788 .38445 .30645 .3761 Metetra
.503 .759 .988 r
.30645 .3761 .31788 .38445 .36524 .37567 .35384 .37045 Metetra
.501 .831 .999 r
.35384 .37045 .36524 .37567 .41373 .36578 .40259 .36072 Metetra
.553 .858 .997 r
.40259 .36072 .41373 .36578 .46336 .35382 .45255 .3474 Metetra
.61 .853 .99 r
.45255 .3474 .46336 .35382 .51409 .33968 .5036 .33137 Metetra
.644 .827 .971 r
.5036 .33137 .51409 .33968 .56587 .3237 .55565 .31355 Metetra
.658 .796 .949 r
.55565 .31355 .56587 .3237 .61866 .30635 .60869 .29467 Metetra
.661 .771 .931 r
.60869 .29467 .61866 .30635 .67245 .28809 .66272 .27527 Metetra
.66 .754 .919 r
.66272 .27527 .67245 .28809 .72729 .26928 .7178 .25569 Metetra
.657 .743 .913 r
.7178 .25569 .72729 .26928 .7832 .25012 .77397 .2361 Metetra
.484 .355 .629 r
.12151 .34511 .1276 .40159 .17203 .38377 .16189 .35874 Metetra
.418 .442 .779 r
.16189 .35874 .17203 .38377 .21589 .37901 .20388 .37173 Metetra
.181 .615 .942 r
.20388 .37173 .21589 .37901 .26051 .37818 .24798 .37867 Metetra
.007 0 0 r
.24798 .37867 .26051 .37818 .30645 .3761 .29411 .37799 Metetra
.117 .666 .705 r
.29411 .37799 .30645 .3761 .35384 .37045 .34195 .37032 Metetra
.367 .834 .886 r
.34195 .37032 .35384 .37045 .40259 .36072 .39119 .35727 Metetra
.564 .894 .985 r
.39119 .35727 .40259 .36072 .45255 .3474 .44157 .34062 Metetra
.645 .858 .981 r
.44157 .34062 .45255 .3474 .5036 .33137 .49295 .32182 Metetra
.667 .81 .952 r
.49295 .32182 .5036 .33137 .55565 .31355 .54527 .30195 Metetra
.669 .774 .928 r
.54527 .30195 .55565 .31355 .60869 .29467 .59855 .28168 Metetra
.666 .751 .914 r
.59855 .28168 .60869 .29467 .66272 .27527 .65282 .26138 Metetra
.662 .738 .907 r
.65282 .26138 .66272 .27527 .7178 .25569 .70814 .24122 Metetra
.657 .731 .904 r
.70814 .24122 .7178 .25569 .77397 .2361 .76457 .22127 Metetra
.542 0 0 r
.10923 .33738 .12151 .34511 .16189 .35874 .14753 .36911 Metetra
.551 .078 .085 r
.14753 .36911 .16189 .35874 .20388 .37173 .18935 .38678 Metetra
.439 0 .077 r
.18935 .38678 .20388 .37173 .24798 .37867 .23415 .39083 Metetra
.272 0 0 r
.23415 .39083 .24798 .37867 .29411 .37799 .28118 .3841 Metetra
.28118 .3841 .29411 .37799 .34195 .37032 .32984 .37002 Metetra
.497 .9 .922 r
.32984 .37002 .34195 .37032 .39119 .35727 .37968 .35162 Metetra
.657 .883 .981 r
.37968 .35162 .39119 .35727 .44157 .34062 .43049 .33109 Metetra
.681 .812 .946 r
.43049 .33109 .44157 .34062 .49295 .32182 .48217 .3098 Metetra
.678 .767 .918 r
.48217 .3098 .49295 .32182 .54527 .30195 .53475 .28849 Metetra
.672 .744 .904 r
.53475 .28849 .54527 .30195 .59855 .28168 .58826 .26744 Metetra
.666 .732 .899 r
.58826 .26744 .59855 .28168 .65282 .26138 .64277 .24673 Metetra
.661 .728 .899 r
.64277 .24673 .65282 .26138 .70814 .24122 .69834 .22629 Metetra
.657 .726 .9 r
.69834 .22629 .70814 .24122 .76457 .22127 .75503 .20608 Metetra
.579 .243 .364 r
.09029 .37737 .10923 .33738 .14753 .36911 .12962 .40573 Metetra
.5 .151 .305 r
.12962 .40573 .14753 .36911 .18935 .38678 .1733 .41233 Metetra
.383 0 .165 r
.1733 .41233 .18935 .38678 .23415 .39083 .21989 .40377 Metetra
.101 0 0 r
.21989 .40377 .23415 .39083 .28118 .3841 .26828 .38623 Metetra
.478 .897 .853 r
.26828 .38623 .28118 .3841 .32984 .37002 .31783 .36433 Metetra
.692 .887 .97 r
.31783 .36433 .32984 .37002 .37968 .35162 .36822 .34099 Metetra
.698 .793 .923 r
.36822 .34099 .37968 .35162 .43049 .33109 .41936 .31775 Metetra
.686 .745 .896 r
.41936 .31775 .43049 .33109 .48217 .3098 .4713 .29524 Metetra
.675 .725 .887 r
.4713 .29524 .48217 .3098 .53475 .28849 .52409 .27353 Metetra
.668 .72 .888 r
.52409 .27353 .53475 .28849 .58826 .26744 .57784 .25248 Metetra
.663 .722 .893 r
.57784 .25248 .58826 .26744 .64277 .24673 .63259 .23183 Metetra
.659 .726 .899 r
.63259 .23183 .64277 .24673 .69834 .22629 .68843 .21138 Metetra
.656 .729 .903 r
.68843 .21138 .69834 .22629 .75503 .20608 .74539 .19098 Metetra
.497 .192 .38 r
.06713 .44312 .09029 .37737 .12962 .40573 .11059 .44655 Metetra
.393 .051 .262 r
.11059 .44655 .12962 .40573 .1733 .41233 .15757 .43086 Metetra
.1 0 0 r
.15757 .43086 .1733 .41233 .21989 .40377 .20633 .40593 Metetra
.612 .956 .888 r
.20633 .40593 .21989 .40377 .26828 .38623 .25597 .3781 Metetra
.735 .841 .93 r
.25597 .3781 .26828 .38623 .31783 .36433 .30616 .35072 Metetra
.707 .741 .878 r
.30616 .35072 .31783 .36433 .36822 .34099 .35687 .32508 Metetra
.686 .704 .861 r
.35687 .32508 .36822 .34099 .41936 .31775 .40821 .30136 Metetra
.673 .696 .863 r
.40821 .30136 .41936 .31775 .4713 .29524 .46032 .27919 Metetra
.666 .702 .873 r
.46032 .27919 .4713 .29524 .52409 .27353 .51331 .25806 Metetra
.662 .713 .886 r
.51331 .25806 .52409 .27353 .57784 .25248 .56728 .23751 Metetra
.659 .724 .897 r
.56728 .23751 .57784 .25248 .63259 .23183 .6223 .21716 Metetra
.657 .732 .905 r
.6223 .21716 .63259 .23183 .68843 .21138 .67841 .19677 Metetra
.656 .737 .91 r
.67841 .19677 .68843 .21138 .74539 .19098 .73567 .17621 Metetra
.345 .029 .278 r
.04476 .49748 .06713 .44312 .11059 .44655 .0938 .4656 Metetra
.0938 .4656 .11059 .44655 .15757 .43086 .14395 .42771 Metetra
.799 .925 .919 r
.14395 .42771 .15757 .43086 .20633 .40593 .19422 .3912 Metetra
.738 .728 .842 r
.19422 .3912 .20633 .40593 .25597 .3781 .24444 .35883 Metetra
.696 .662 .813 r
.24444 .35883 .25597 .3781 .30616 .35072 .29479 .3308 Metetra
.676 .65 .817 r
.29479 .3308 .30616 .35072 .35687 .32508 .34555 .30627 Metetra
.666 .661 .835 r
.34555 .30627 .35687 .32508 .40821 .30136 .39697 .28408 Metetra
.661 .681 .858 r
.39697 .28408 .40821 .30136 .46032 .27919 .44921 .26326 Metetra
.659 .703 .88 r
.44921 .26326 .46032 .27919 .51331 .25806 .50239 .24306 Metetra
.657 .721 .896 r
.50239 .24306 .51331 .25806 .56728 .23751 .5566 .22304 Metetra
.657 .734 .907 r
.5566 .22304 .56728 .23751 .6223 .21716 .61189 .20289 Metetra
.656 .742 .913 r
.61189 .20289 .6223 .21716 .67841 .19677 .66829 .18246 Metetra
.656 .746 .916 r
.66829 .18246 .67841 .19677 .73567 .17621 .72584 .16169 Metetra
.857 .997 .805 r
.02861 .50498 .04476 .49748 .0938 .4656 .08161 .44728 Metetra
.77 .691 .776 r
.08161 .44728 .0938 .4656 .14395 .42771 .13307 .39998 Metetra
.696 .594 .738 r
.13307 .39998 .14395 .42771 .19422 .3912 .18342 .36292 Metetra
.666 .576 .744 r
.18342 .36292 .19422 .3912 .24444 .35883 .23334 .3337 Metetra
.655 .593 .773 r
.23334 .3337 .24444 .35883 .29479 .3308 .28343 .30953 Metetra
.653 .628 .812 r
.28343 .30953 .29479 .3308 .34555 .30627 .33408 .2882 Metetra
.654 .667 .85 r
.33408 .2882 .34555 .30627 .39697 .28408 .38554 .26818 Metetra
.655 .701 .88 r
.38554 .26818 .39697 .28408 .44921 .26326 .43793 .24858 Metetra
.656 .725 .9 r
.43793 .24858 .44921 .26326 .50239 .24306 .49133 .22896 Metetra
.656 .738 .911 r
.49133 .22896 .50239 .24306 .5566 .22304 .54579 .20908 Metetra
.656 .746 .916 r
.54579 .20908 .5566 .22304 .61189 .20289 .60135 .18884 Metetra
.656 .749 .918 r
.60135 .18884 .61189 .20289 .66829 .18246 .65804 .16821 Metetra
.656 .75 .92 r
.65804 .16821 .66829 .18246 .72584 .16169 .71589 .14717 Metetra
.677 .509 .65 r
.02099 .45459 .02861 .50498 .08161 .44728 .0737 .39762 Metetra
.637 .481 .655 r
.0737 .39762 .08161 .44728 .13307 .39998 .12371 .35977 Metetra
.624 .498 .688 r
.12371 .35977 .13307 .39998 .18342 .36292 .17276 .33301 Metetra
.629 .546 .743 r
.17276 .33301 .18342 .36292 .23334 .3337 .22189 .3116 Metetra
.64 .613 .807 r
.22189 .3116 .23334 .3337 .28343 .30953 .27165 .29223 Metetra
.65 .675 .86 r
.27165 .29223 .28343 .30953 .33408 .2882 .32228 .27334 Metetra
.654 .716 .894 r
.32228 .27334 .33408 .2882 .38554 .26818 .37387 .25433 Metetra
.655 .737 .91 r
.37387 .25433 .38554 .26818 .43793 .24858 .42647 .23501 Metetra
.656 .746 .917 r
.42647 .23501 .43793 .24858 .49133 .22896 .48012 .21532 Metetra
.656 .75 .919 r
.48012 .21532 .49133 .22896 .54579 .20908 .53484 .19523 Metetra
.656 .751 .92 r
.53484 .19523 .54579 .20908 .60135 .18884 .59067 .17474 Metetra
.656 .751 .92 r
.59067 .17474 .60135 .18884 .65804 .16821 .64764 .15382 Metetra
.656 .751 .92 r
.64764 .15382 .65804 .16821 .71589 .14717 .7058 .13248 Metetra
.548 .367 .588 r
.01927 .36806 .02099 .45459 .0737 .39762 .06551 .35093 Metetra
.581 .449 .666 r
.06551 .35093 .0737 .39762 .12371 .35977 .11262 .33348 Metetra
.621 .565 .77 r
.11262 .33348 .12371 .35977 .17276 .33301 .16061 .31571 Metetra
.646 .669 .858 r
.16061 .31571 .17276 .33301 .22189 .3116 .20952 .29759 Metetra
.654 .724 .901 r
.20952 .29759 .22189 .3116 .27165 .29223 .25937 .27913 Metetra
.655 .744 .915 r
.25937 .27913 .27165 .29223 .32228 .27334 .31019 .26031 Metetra
.656 .75 .919 r
.31019 .26031 .32228 .27334 .37387 .25433 .362 .24112 Metetra
.656 .751 .92 r
.362 .24112 .37387 .25433 .42647 .23501 .41485 .22155 Metetra
.656 .751 .92 r
.41485 .22155 .42647 .23501 .48012 .21532 .46874 .20159 Metetra
.656 .751 .92 r
.46874 .20159 .48012 .21532 .53484 .19523 .52373 .18122 Metetra
.656 .751 .92 r
.52373 .18122 .53484 .19523 .59067 .17474 .57984 .16044 Metetra
.656 .751 .92 r
.57984 .16044 .59067 .17474 .64764 .15382 .6371 .13923 Metetra
.656 .751 .92 r
.6371 .13923 .64764 .15382 .7058 .13248 .69556 .11758 Metetra
0 g
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.18558 .19702 m
.19014 .20174 L
s
[(0.25)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(0.5)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(0.75)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(1)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.1552 .20892 m
.15797 .21172 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.27893 .16047 m
.28157 .16339 L
s
.3108 .14799 m
.31341 .15094 L
s
.37572 .12257 m
.37826 .12558 L
s
.40878 .10962 m
.41128 .11266 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.54521 .0562 m
.54755 .05937 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[10]:=",
  ImageSize->{288, 233.562},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jB000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00a@3oool5000005H0oooo00370?ooo`030000003o
ool0oooo05H0oooo00370?ooo`030000003oool0oooo05H0oooo00370?ooo`030000003oool0oooo
05H0oooo00370?ooo`030000003oool0oooo05H0oooo00360?ooo`800000F03oool00=00oooo0P00
001>0?ooo`00c`3oool010000000oooo0?ooo`00001=0?ooo`00c`3oool010000000oooo0?ooo`00
001=0?ooo`00c`3oool010000000oooo0?ooo`00001=0?ooo`00c`3oool010000000oooo0?ooo`00
001=0?ooo`00bP3oool2000000@0oooo0P00001>0?ooo`00b03oool5000005<0oooo00350?ooo`<0
00000`3oool2000005<0oooo00330?ooo`8000001P3oool00`000000oooo0000001B0?ooo`00`03o
ool3000000P0oooo00@000000?ooo`3oool00000D@3oool00;d0oooo0`0000000`3oool000000?oo
o`080?ooo`040000003oool0oooo00000540oooo002k0?ooo`8000003P3oool01@000000oooo0?oo
o`3oool000000500oooo002h0?ooo`<00000403oool010000000oooo0?ooo`3oool2000004l0oooo
00260?ooo`800000103oool2000000D0oooo00<000000?ooo`3oool00P3oool2000001`0oooo0P00
000C0?ooo`060000003oool0oooo0000003oool00000C`3oool008D0oooo00@000000?ooo`3oool0
00002P3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool05P3oool30000
0080oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
04`0oooo00250?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`0D0?ooo`800000603oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
04`0oooo00250?ooo`040000003oool0oooo000000/0oooo00@000000?ooo`3oool0oooo0P00000D
0?ooo`<000006P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo04/0oooo00250?oo
o`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?oo
o`0A0?ooo`<000007@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo04X0oooo0026
0?ooo`8000002@3oool4000000<0oooo0`00000>0?ooo`<00000803oool00`000000oooo0?ooo`04
0?ooo`030000003oool0oooo04X0oooo002V0?ooo`<000008`3oool00`000000oooo0?ooo`040?oo
o`800000B`3oool00:@0oooo0P00000V0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3o
ool0B03oool00:40oooo0`0000000`3oool000000?ooo`0U0?ooo`030000003oool0oooo00H0oooo
00<000000?ooo`3oool0B03oool009l0oooo0P00000[0?ooo`030000003oool0oooo00L0oooo00<0
00000?ooo`3oool0A`3oool009`0oooo0`00000]0?ooo`030000003oool0oooo00P0oooo00<00000
0?ooo`3oool0AP3oool009X0oooo0P00000`0?ooo`030000003oool0oooo00P0oooo00<000000?oo
o`3oool0AP3oool009L0oooo0`0000020?ooo`030000003oool0oooo02d0oooo00<000000?ooo`3o
ool02@3oool00`000000oooo0?ooo`150?ooo`00U03oool3000003D0oooo00<000000?ooo`3oool0
203oool3000004H0oooo002B0?ooo`800000>03oool00`000000oooo0?ooo`0:0?ooo`030000003o
ool0oooo04@0oooo002?0?ooo`<0000000<0oooo0000003oool0=`3oool00`000000oooo0?ooo`0;
0?ooo`030000003oool0oooo04<0oooo002=0?ooo`800000?@3oool00`000000oooo0?ooo`0<0?oo
o`030000003oool0oooo0480oooo002:0?ooo`<00000?`3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0oooo0480oooo001M0?ooo`800000103oool2000000@0oooo0P00000M0?ooo`800000
@P3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo0440oooo001L0?ooo`040000003o
ool0oooo000000P0oooo00@000000?ooo`3oool000006@3oool300000080oooo00<000000?ooo`3o
ool0?`3oool00`000000oooo0?ooo`0=0?ooo`800000@P3oool005`0oooo00@000000?ooo`3oool0
00002`3oool00`000000oooo0?ooo`0E0?ooo`800000A`3oool00`000000oooo0?ooo`0<0?ooo`03
0000003oool0000000L0oooo1@00000f0?ooo`00G03oool010000000oooo0?ooo`0000090?ooo`80
00005@3oool3000004T0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`060?ooo`03
0000003oool0oooo03H0oooo001L0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3o
ool04@3oool3000000030?ooo`000000oooo04T0oooo00<000000?ooo`3oool03`3oool00`000000
oooo0?ooo`060?ooo`030000003oool0oooo03H0oooo001M0?ooo`8000002P3oool3000000l0oooo
0P00001?0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool01@3oool00`000000oooo
0?ooo`0f0?ooo`00N03oool300000500oooo0P00000C0?ooo`030000003oool0oooo00@0oooo00<0
00000?ooo`3oool0=P3oool007H0oooo0P00001@0?ooo`<0000000<0Y<;^0000000000004P3oool0
0`000000oooo0?ooo`030?ooo`800000>03oool007<0oooo0`0000020?ooo`030000003oool0oooo
04X0oooo0`0000001P2/`^h0Y;k^0:c2kP2T_^h0[<;^00000140oooo0`00000n0?ooo`00L@3oool2
000004l0oooo0`000000202T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0P00000C0?ooo`03
0000003oool0oooo03/0oooo001^0?ooo`<00000CP3oool30000000;0:c2kP2T_^h0[<;^0:BnkP2/
`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h00`00000B0?ooo`030000003oool0oooo03/0oooo001/
0?ooo`80000000<0oooo0000003oool0B`3oool30000000<0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV
0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0`0000000`2T_^H0Y<;^0000000B0?ooo`030000003oool0
oooo03X0oooo001Y0?ooo`<00000103oool00`000000oooo0?ooo`170?ooo`80000000`0[<;^0:Bn
kP000000Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h3000000070:BnkP2/`^h00000
0:c2kP2T_^h0[<;^0000000B0?ooo`030000003oool0oooo03T0oooo001V0?ooo`<00000CP3oool3
0000000;0:BniP2T`^h0Y;kV0:C2kP000000Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H00`0000002@2T
_^H0Y<;^0:BniP2T`^h0Y;kV0000002T_^H0Y<;^0:BniP0200000140oooo00<000000?ooo`3oool0
>@3oool006@0oooo1000001=0?ooo`80000000/0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2
kP000000[<;^0:BnkP030000000:0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h000000:c2
kP80000000<0Y;k^0000003oool03`3oool3000003X0oooo001Q0?ooo`<00000CP3oool30000000:
0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP<0000000X0Y;kV0:C2kP2T_^H0
Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0`0000001@2T_^h0Y;kV0:BnkP2T_^H000000180oooo
00<000000?ooo`3oool0=`3oool002d0oooo0P0000040?ooo`8000000`3oool4000000<0oooo0P00
000N0?ooo`800000C`3oool20000000;0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^
0:BnkP2/`^h00P0000002P2T_^h000000:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h30000
00080:c2kP000000[<;^0:BnkP2/`^h0Y;k^0:c2kP00000B0?ooo`030000003oool0oooo03L0oooo
000/0?ooo`040000003oool0oooo000000P0oooo00<000000?ooo`3oool00`3oool010000000oooo
0?ooo`00000J0?ooo`<00000CP3oool30000000:0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T
`^h0Y;kV0:C2kP<0000000X0Y<;^0:BniP2T`^h0Y;kV0000002T_^H0Y<;^0:BniP2T`^h0Y;kV0P00
00002P2T_^h0Y;kV0:BnkP2T_^H000000:BniP2T_^h0Y;kV0:BnkP2T_^H200000180oooo00<00000
0?ooo`3oool0=P3oool002`0oooo00@000000?ooo`3oool000002@3oool00`000000oooo0?ooo`05
0?ooo`030000003oool0oooo01H0oooo0P0000020?ooo`030000003oool0oooo04X0oooo0P000000
2P2T_^h000000:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h30000000:0:c2kP2T_^h0[<;^
0:BnkP2/`^h0Y;k^0:c2kP000000[<;^0:BnkP<0000000X0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0
00000:c2kP2T_^h0[<;^0P0000000`2T_^H0[;k^0000000B0?ooo`030000003oool0oooo03D0oooo
000/0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3oool2000001H0oooo
0`00001>0?ooo`<0000000X0Y<;^0:BniP2T`^h0Y;kV0000002T_^H0Y<;^0:BniP2T`^h0Y;kV0P00
00002P2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H3000000090:BniP2T_^h0
Y;kV0:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV00@0000000D0Y;k^0:BniP2T_^h0Y;kV0000000B0?oo
o`030000003oool0oooo03D0oooo000/0?ooo`040000003oool0oooo000000P0oooo00@000000?oo
o`3oool000000`3oool00`000000oooo0?ooo`0C0?ooo`800000C`3oool20000000:0:c2kP2T_^h0
[<;^0:BnkP2/`^h0Y;k^0:c2kP000000[<;^0:BnkP<0000000X0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/
`^h0Y;k^0:c2kP2T_^h0[<;^0P0000002`2T_^h0[<;^0000002/`^h0Y;k^0:c2kP2T_^h0[<;^0:Bn
kP2/`^h0Y;k^0080000000T0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP000000403oool3
000003H0oooo000]0?ooo`8000002P3oool2000000@0oooo0`00000@0?ooo`<0000000<0oooo0000
003oool0B`3oool3000000090:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV00<00000
00X0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0`0000002P2T_^h0Y;kV0:Bn
kP2T_^H000000:BniP2T_^h0Y;kV0:BnkP2T_^H30000000:0:BniP000000Y;kV0:BnkP2T_^H0Y;k^
0:BniP2T_^h0Y;kV0:BnkP8000004P3oool00`000000oooo0?ooo`0c0?ooo`00C`3oool3000004l0
oooo0P0000002@2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP030000000;0:BnkP2/
`^h000000:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h00P0000002`2/`^h0Y;k^0:c2kP2T
_^h0[<;^0:BnkP2/`^h0Y;k^0000002T_^h0[<;^0080000000/0Y;kV0:bnkP2T_^H0[;k^0000002/
_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP02000000030:bnkP2T^^H000000140oooo00<000000?ooo`3o
ool0<`3oool004d0oooo0P00001?0?ooo`<0000000T0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^
0:BniP2T`^h00P0000002`2T_^H0Y<;^0:BniP2T`^h0Y;kV0000002T_^H0Y<;^0:BniP2T`^h0Y;kV
00<0000000X0Y;kV0:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^0`0000002P2T_^h0
Y;kV0:BnkP2T_^H0Y;k^0:BniP000000Y;kV0:BnkP2T_^H3000000050:BjiP2T_^H0Y;[V0:BniP00
00004P3oool00`000000oooo0?ooo`0b0?ooo`00BP3oool5000004`0oooo0`0000002@2/`^h0Y;k^
0000002T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP030000000;0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^
0:c2kP2T_^h000000:BnkP2/`^h00P0000002P2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^
0:BnkP2/`^h30000000;0:bnkP2T_^H000000:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0000002/_^h0
0P0000002@2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0000000A0?ooo`030000003oool0
oooo0380oooo00180?ooo`800000CP3oool3000000090:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0000
002T_^H0Y<;^00<0000000/0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2
kP03000000090:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^00<0000000/0Y;k^0:Bj
iP2T_^h0Y;[V0:BnkP000000Y;k^0:BniP2T_^h0Y;kV0:BnkP030000000:0:BniP2T^^H0Y;kV0:Bj
iP2T_^H0Y;[V0:BniP2T^^H0Y;kV0:BjiP8000004@3oool00`000000oooo0?ooo`0a0?ooo`00A@3o
ool3000004d0oooo0`0000002@2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP030000
000;0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h00`0000002@2/_^h0
Y;kV0000002T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP030000000<0:bnkP2T_^H0[;k^0:BniP2/_^h0
Y;kV0:bnkP2T_^H0[;k^0000002/_^h0Y;kV0P0000002`2/_^h000000:bnkP2T^^H0[;k^0:BjiP2/
_^h0Y;[V0:bnkP2T^^H0[;k^0080000000<0Y;[V0000003oool03`3oool300000380oooo00130?oo
o`<00000C03oool30000000:0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP80
000000/0Y;k^0000002T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP03000000090:Bn
kP2T_^H0Y;k^0:BniP2T_^h0Y;kV0000002T_^H0Y;k^00<0000000`0Y;k^0:BjiP2T_^h0Y;[V0:Bn
kP2T^^H0Y;k^0:BjiP2T_^h0Y;[V0:BnkP2T^^H30000000:0:BjiP2T_^H0Y;[V0000002T^^H0Y;kV
0:BjiP2T_^H0Y;[V0:BniP<00000102T^^H00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo
0300oooo00100?ooo`<000000`3oool00`000000oooo0?ooo`160?ooo`<0000000X0[<;^0:BnkP2/
`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0`0000002P2T_^h0[<;^0:BnkP2/`^h000000:c2
kP2T_^h0[<;^0:BnkP2/`^h3000000090:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^
00<0000000`0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H30000
000:0:BjiP000000Y;[V0:bnkP2T^^H0[;k^0000002/_^h0Y;[V0:bnkP<0000000P0[;[^0:BjiP2/
^^h0Y;[V0:bjkP2T^^H0[;[^00000180oooo00<000000?ooo`3oool0;`3oool003d0oooo0`00001<
0?ooo`<0000000/0Y<;^0:BniP000000Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP020000
000:0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^0:BniP000000Y;kV0:C2kP<0000000X0Y;k^0:BniP2T
_^h0Y;kV0:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV0P000000302/^^H0Y;[V0000002T^^H0Y;k^0:Bj
iP2T_^h0Y;[V0:BnkP2T^^H0Y;k^0:BjiP<0000000/0Y;[V0:bjiP2T^^H0[;[V0:BjiP000000Y;[V
0:BniP2T^^H000000:BjiP02000000X0Y;[V0P00000B0?ooo`030000003oool0oooo02h0oooo000k
0?ooo`<00000C03oool20000000;0:c2kP2T_^h0[<;^0:BnkP2/`^h000000:c2kP2T_^h0[<;^0:Bn
kP2/`^h00`0000002@2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP030000000:0:bn
kP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP<0000000/0Y;[V0:bnkP2T^^H0[;k^
0000002/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP030000000;0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^
0:BjiP2/_^h000000:bnkP2T^^H00`0000002@2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^
0:BjiP03000000030:BjiP2/^^H000000140oooo00<000000?ooo`3oool0;P3oool003P0oooo0`00
00020?ooo`030000003oool0oooo04L0oooo0`0000002`2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T
_^H0Y<;^0000002T`^h0Y;kV0080000000T0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^0:BniP2T
_^h00`0000002`2T_^h0Y;[V0000002T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^00800000
00/0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP000000Y;k^0:BjiP030000000;0:BjiP2/
^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H00`0000020:BjiP030000002T^^H0
Y;[V00@0Y;[V0`0000001P2T]^H0[;[V0:BfiP2/^^H0Y;KV00000180oooo00<000000?ooo`3oool0
;@3oool001L0oooo0P00000M0?ooo`800000C03oool30000000;0:BnkP2/`^h0Y;k^0:c2kP2T_^h0
[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h00`0000002@2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0
[<;^0:BnkP020000000;0:bnkP2T^^H0[;k^0:BjiP2/_^h000000:bnkP2T_^H0[;k^0:BniP2/_^h0
0`0000002P2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H30000000;0:BjiP2/
_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H00`0000002@2T^^H0[;[V0000002/
^^h0Y;[V0000002T^^H0[;[^0:BjiP030000000:0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/
^^H0Y;[V00000100oooo0`0000060?ooo`@00000903oool001H0oooo00@000000?ooo`3oool00000
6@3oool3000004/0oooo0`0000002`2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T
`^h0Y;kV00<0000000T0Y;kV0:BnkP000000Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H00`0000002`2T
^^H0Y;k^0:BjiP2T_^h0Y;[V0:BnkP2T^^H0Y;k^0000002T_^h0Y;kV0080000000X0[;[V0:BjiP2/
^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0`000000302T]^H0[;[V0000002/^^H0Y;[V0:bj
iP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP80000000P0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H00000
0:BjiP@0000000`0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H2
00000140oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`0U0?ooo`005P3oool01000
0000oooo0?ooo`00000G0?ooo`8000000P3oool00`000000oooo0?ooo`160?ooo`<0000000`0Y;k^
0:c2kP000000[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h2000000090:BniP2/_^h0
Y;kV0:bnkP2T_^H0[;k^0000002/`^h0Y;k^00<0000000/0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0
Y;[V0:bnkP2T^^H0[;k^0:BjiP03000000090:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0
Y;[V00<0000000`0Y;[V0:bjiP2T^^H0[;[V0:BjiP000000Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h3
000000080:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H30000000<0:BjiP2/^^H0Y;[V0:bj
iP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0`0000000`2/_^h000000?ooo`0A0?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool0903oool001H0oooo00@000000?ooo`3oool0
0000503oool3000004/0oooo0P000000302T_^H0Y<;^0:BniP2T`^h0Y;kV0000002T_^H0Y<;^0:Bn
iP2T`^h0Y;kV0:C2kP<0000000P0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^0:BniP<0000000/0
Y;[V0:BnkP2T^^H0Y;k^0:BjiP2T_^h0Y;[V0:BnkP2T^^H0Y;k^0:BjiP03000000090:BfiP2/^^H0
00000:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V00<0000000`0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H0
Y;KV0:bjiP2T]^H000000:BjiP2/^^H3000000090:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T
]^H0[;[V0080000000`0Y;[V0:bjiP000000[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/
^^H3000000060:bjiP2T^^H0[;[V0:BjiP2/^^H000004P3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo02<0oooo000F0?ooo`040000003oool0oooo00000180oooo0P00001;0?ooo`<0
000000/0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP000000Y;k^0:c2kP03000000080:bn
kP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H30000000<0:BjiP2/^^H000000:bnkP2T^^H0[;k^
0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0P0000002@2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP000000
[;k^0:BjiP030000000<0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0
[;[V0`0000002@2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP030000000;0:bjiP2T
^^H0[;[V0:BjiP000000Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H00`0000002@2/_^h0Y;[V0:bnkP2T
^^H0[;k^0:BjiP2/_^h0Y;[V0000000B0?ooo`030000003oool0oooo0080oooo00@000000?ooo`3o
ool00000903oool001L0oooo0P00000@0?ooo`<0000000<0oooo0000003oool0A`3oool30000000;
0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h00`0000002@2T_^h0Y;kV
0:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP020000000<0:BfiP2T^^H0Y;KV0:BjiP2T]^H00000
0:BjiP2T_^h0Y;[V0:BnkP2T^^H0Y;k^0`000000202/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H0
Y;KV0`000000302T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP<0
000000X0[;[V0000002/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0P0000001`2/^^H0Y;[V
0:bjiP2T^^H0[;[V0:BjiP2/^^H00P0000000`2T]^H0[;[V000000020000000;0:bjiP2T^^H0[;[V
0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H00P00000B0?ooo`030000003oool0oooo0080
oooo0P00000U0?ooo`009P3oool3000004X0oooo0`0000002`2/`^h0Y;k^0:c2kP2T_^h0[<;^0:Bn
kP2/`^h0Y;k^0:c2kP2T_^h0[<;^00<0000000T0[;k^0:BjiP000000Y;kV0:bnkP2T_^H0[;k^0:Bn
iP2/_^h00`0000002`2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0000002T^^H0[;k^00<0
000000T0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H00P000000302T]]h0[;KV0000
002/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP<0000000X0[;[V0:BfiP2/^^H0Y;KV
0:bjiP000000[;[V0:BjiP2/^^H0Y;[V0`0000002@2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0
[;[V0:BjiP040000000;0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0
0`0000000`2/_^h0Y;kV0000000?0?ooo`@00000:P3oool002@0oooo0P00001;0?ooo`80000000`0
Y;k^0:BniP000000Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H2000000090:BniP2T
^^H0Y;kV0:BjiP2T_^H0Y;[V0000002T_^H0Y;k^00<0000000/0Y;[V0:BfiP2T^^H0Y;KV0:BjiP2T
]^H0Y;[V0:BfiP2T^^H0Y;KV0:BjiP030000000:0:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T
]^H0[;[V0:BfiP80000000/0[;KV0:BfgP2/]^H0Y;KN0000002T]^H0[;[V0:BfiP2/^^H0Y;KV0:bj
iP030000000;0:bjiP2T]]h0[;[V0:BfgP2/^^H0Y;KN0:bjiP2T]]h000000:BfiP2/^^H00P000000
2`2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0080000000/0[;kV0000
002/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP03000000060:bnkP2T_^H0[;k^0:Bn
iP2/_^h000004P3oool00`000000oooo0?ooo`0X0?ooo`008@3oool300000080oooo00<000000?oo
o`3oool0AP3oool20000000;0:BnkP2/`^h0Y;k^0:c2kP000000[<;^0:BnkP2/`^h0Y;k^0:c2kP2T
_^h00`000000202T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0`0000002`2/^^H0Y;[V0:bj
iP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V00<0000000X0[;KV0:BbgP000000Y;KV0:bj
iP2T]^H0[;[V0:BfiP2/^^H0Y;KV0`0000002P2T]]h0[;KV0:BfgP2/]^H0Y;KN0:bfiP2T]]h00000
0:BjiP2/^^H30000000;0:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H0
0`0000002P2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H5000000080:BjiP2/
_^h000000:bnkP2T^^H0[;k^0:BjiP2/_^h30000000:0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bn
kP2T_^H0[;k^00000180oooo00<000000?ooo`3oool09`3oool001l0oooo0P00001;0?ooo`<00000
00X0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^0000002T`^h0Y;kV0`0000002@2T^^H0Y;kV0:Bj
iP2T_^H0Y;[V0:BniP2T^^H0Y;kV0:BjiP020000000<0:BbiP2T/]h000000:BfiP2T^^H0Y;KV0:Bj
iP2T]^H0Y;[V0:BfiP2T^^H0Y;KV0P0000002`2//^H0Y;;N0:bbiP2T/]h0[;;V0000002/^^H0Y;KV
0:bjiP2T]^H0[;[V0080000000X0Y;KN0:bfiP2T]]h0[;KV0:BfgP2/]^H0Y;KN0:bfiP2T]]h0[;KV
0`0000002`2/^^H0Y;KN0:bjiP2T]]h0[;[V0:BfgP2/^^H0Y;KN0:bjiP2T]]h0[;[V00<0000000/0
[;[V0:BjiP000000Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP05000000070:bniP2T^^H0
00000:BjiP000000Y;[V0:bjiP030000000<0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0
[;k^0:BniP2/_^h0Y;kV0P00000A0?ooo`030000003oool0oooo02L0oooo000M0?ooo`800000B`3o
ool20000000:0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP<0000000X0Y;[V
0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0P0000002`2T/]h0[;KV0:BbgP2/]^H0
00000:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V00<0000000/0Y;;N0:bfiP2T/]h0[;KV0:BbgP2/]^H0
Y;;N0:bfiP000000[;[V0:BfiP020000000:0:bfiP2T]]h0[;KV0:BfgP2/]^H0Y;KN0:bfiP2T]]h0
[;KV0:BfgP80000000`0[;[V0:bfiP000000Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T
]^H20000000;0:bniP2/^^H0[;kV0:bjiP2/_^H000000:bjiP2T^^H0[;[V0:BjiP2/^^H01@000000
1`2/_^h0Y;[V0:bnkP2T^^H0[;k^0000002/_^h00`000000302/_^h0Y;kV0:bnkP2T_^H0[;k^0:Bn
iP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP<0000000<0Y<;^0:c2kP0000004@3oool00`000000oooo
0?ooo`0V0?ooo`007@3oool3000004P0oooo0P0000002@2T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T
_^H0Y;k^0:BniP030000000:0:BfgP2T]^H000000:BniP2T^^H0Y;kV0:BjiP2T_^H0Y;[V0:BniP<0
000000X0Y;;V0:BbgP2T/^H0Y;;N0:BbiP2T/]h0Y;;V0000002T^^H0Y;KV0`0000002`2T/]h0[;;V
0:BbgP2//^H0Y;;N0:bbiP2T/]h0[;;V0:BbgP2//^H0Y;;N00<0000000X0Y;KN0:bfiP2T]]h0[;KV
0:BfgP2/]^H0Y;KN0:bfiP2T]]h0[;KV0P0000002`2T]]h0[;[V0:BfgP2/^^H000000:bjiP2T]]h0
[;[V0:BfgP2/^^H0Y;KN00<0000000/0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP000000
[;[V0:BjiP04000000080:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H20000000<0:c2kP2T
_^H000000:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0`0000001P2T`^H0Y<;^0:C2
iP2T`^h0Y<;V00000100oooo0`00000W0?ooo`007@3oool3000004D0oooo0`0000002@2/`^h0Y;k^
0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP020000000;0:BfiP2/^^H0Y;KV0:bjiP2T]^H00000
0:BjiP2/_^h0Y;[V0:bnkP2T^^H00P0000002P2/]^H0Y;;N0:bfiP2T/]h0[;KV0:BbgP2/]^H0Y;;N
0:bfiP2T/]h30000000<0:BbgP2/]^H0Y;;N0:bfiP2T/]h0[;KV0:BbgP2/]^H0Y;;N0:bfiP2T/]h0
[;KV0P0000002P2//]h0[;KV0000002/]^H0Y;KN0:bfiP2T]]h0[;KV0:BfgP2/]^H30000000:0:bj
iP2/]^H0[;[V0:bfiP2/^^H0[;KV0:bjiP000000[;[V0:BfiP<0000000/0[;[V0:bniP2/^^H0[;kV
0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP04000000090:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V
0:bnkP2T^^H0[;k^0080000000/0Y<;^0:c2kP000000[<;^0000002/_^h0Y;kV0:bnkP2T_^H0[;k^
0:BniP03000000080:C2kP2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2kP2/`^h200000140oooo00<00000
0?ooo`3oool09@3oool001D0oooo1@0000030?ooo`040000003oool0oooo00000440oooo0`000000
2@2T_^h0Y;kV0000002T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP030000000;0:BfiP2T]]h0Y;KV0:Bf
gP2T]^H0Y;KN0:BfiP2T]]h000000:BjiP2T_^H00P0000002P2T/]h0Y;;V0:BbgP2T/^H0Y;;N0:Bb
iP2T/]h0Y;;V0:BbgP2T/^H20000000=0:B^e@2/[]h000000:bbiP2T/]h0[;;V0:BbgP2//^H0Y;;N
0:bbiP2T/]h0[;;V0:BbgP020000000:0:bbgP2T/]h0[;;N0:BbgP000000Y;KN0:bfiP2T]]h0[;KV
0:BfgP80000000X0[;[V0:BfgP2/^^H0Y;KN0:bjiP2T]]h0[;[V0:BfgP2/^^H0Y;KN0`0000002`2T
^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V00H0000000L0[;kV0:BjiP2/
_^H0Y;[V0:bniP2T^^H0[;kV00<0000000X0[<;^0:BniP2/`^h0Y;kV0:c2kP000000[<;^0000002/
_^h0Y;kV0`000000202T`^H0Y<;^0:C2iP2T`^h0Y<;V0:C2kP2T`^H0Y<;^0`0000000`2Ta^h00000
0?ooo`0A0?ooo`030000003oool0oooo02@0oooo000G0?ooo`030000003oool0oooo0080oooo0`00
00020?ooo`030000003oool0oooo03`0oooo0P0000002@2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP00
0000Y;k^0:c2kP030000000;0:bjiP2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/
^^H00`0000002P2/]^H0Y;;N0:bfiP2T/]h0[;KV0:BbgP2/]^H0Y;;N0:bfiP2T/]h20000000<0:b^
gP2T[]h0[:kN0:B^gP000000Y;;N0:bfiP2T/]h0[;KV0:BbgP2/]^H0Y;;N0`0000002P2//]h0[;KV
0:bbgP2/]^H0[;;N0:bfiP2//]h000000:BfgP2/]^H20000000:0:bfiP2/^^H0[;KV0:bjiP2/]^H0
[;[V0:bfiP2/^^H0[;KV0:bjiP80000000`0[;kV0:bnkP000000[;kV0:bjiP2/_^H0[;[V0:bniP2/
^^H0[;kV0:bjiP2/_^H4000000090:bniP2/`^h0[;kV0000002T^^H0[;k^0:BjiP2/_^h0Y;[V0080
000000X0[<;^0:C2kP2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2kP000000Y<;^0`000000202T`^h0[<;^
0:C2kP2/`^h0Y<;^0:c2kP2T`^h0[<;^0`0000001P2/a_H0Y<;^0:c6mP2T`^h0[<Kf00000180oooo
00<000000?ooo`3oool08`3oool001L0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?oo
o`030?ooo`030000003oool0oooo03P0oooo0`000000202T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T
_^H0Y;k^0`0000002`2T]^H0Y;KN0:BfiP2T]]h0Y;KV0:BfgP2T]^H0Y;KN0:BfiP2T]]h0Y;KV00<0
000000X0Y:[N0:BZe@000000Y;;N0:BbiP2T/]h0Y;;V0:BbgP2T/^H0Y;;N0`000000302T[]D0[:kN
0:B^e@2/[]h0Y:kE0:b^gP2T[]D000000:BbgP2//^H0Y;;N0:bbiP80000000X0Y;;N0:bbgP2T/]h0
[;;N0:BbgP2//]h0Y;;N0:bbgP2T/]h0[;;N0`0000002P2/^^H0Y;KN0:bjiP2T]]h0[;[V0:BfgP2/
^^H0Y;KN0:bjiP2T]]h2000000@0[;kV00L000000:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V00@00000
00X0Y;kV0:c2kP2T_^H0[<;^0:BniP2/`^h0Y;kV0000002T^^H0[;kV0P0000002@2T_^H0[<;^0:Bn
iP2/`^h0Y;kV0:c2kP2T_^H0[<;^0:BniP03000000080:C2kP2Ta^h000000:C2kP2T`^H0Y<;^0:C2
iP2T`^h3000000090:C6kP2T`^h0Y<K^0:C2kP2Ta^h0Y<;^0:C6kP2T`^h000000180oooo00<00000
0?ooo`3oool08`3oool000l0oooo100000040?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`0d0?ooo`<0000000T0Y;kV0:bnkP2T_^H0[;k^0:BniP2/
_^h0Y;kV0:bnkP2T_^H00P000000302/[]h0Y:[N0000002T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T
]^H0[;[V0:BfiP80000000/0[:[N0:BZe@2/Z]h0Y:[E0:bZgP000000[;KV0:BbgP2/]^H0Y;;N0:bf
iP020000000=0:B^gP2/[]h0Y:kN0:b^gP2T[]h0[:kN0:B^gP2/[]h0Y:kN0:b^gP000000[;KV0:Bb
gP020000000:0:bfiP2//]h0[;KV0:bbgP2/]^H0[;;N0:bfiP2//]h0[;KV0:bbgP8000000P2/^^H0
20000000[;KV0:bjiP2/]^H0[;[V0:bfiP2/^^H0[;KV0`0000002P2/_^H0[;k^0:bniP2/_^h0[;kV
0:bnkP2/_^H000000:bjiP2/_^H40000000:0:bniP2/`^h0[;kV0:c2kP2/_^H0[<;^0:bniP2/`^h0
[;kV0:c2kP<0000000P0[<;^0:C2kP2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2kP<0000000P0Y<;^0:c6
kP2T`^h000000:C2kP000000Y<;^0:c2kP<0000000/0[<Kf0:C2kP2/a_H0Y<;^0:c6mP2T`^h0[<Kf
0:C2kP2/a_H0Y<;^0:c6mP0200000180oooo00<000000?ooo`3oool08P3oool001L0oooo00<00000
0?ooo`3oool00P3oool2000000H0oooo00<000000?ooo`3oool0<03oool3000000090:BniP2T_^h0
Y;kV0:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV00<0000000/0Y:[E0:B^gP2TZ]D0Y:kN0000002T]^H0
Y;KN0:BfiP2T]]h0Y;KV0:BfgP030000000;0:BZe@2TZ]h0Y:[E0:BZgP2TZ]D0Y:[N0:BZe@2TZ]h0
00000:BbiP2T/]h00P000000302/[]h0Y:kE0:b^gP2T[]D0[:kN0:B^e@2/[]h0Y:kE0:b^gP2T[]D0
[:kN0:B^e@<0000000X0Y;;N0:bbgP2T/]h0[;;N0:BbgP2//]h0Y;;N0:bbgP2T/]h0[;;N0P000000
2P2/]]h0[;[V0:bfgP2/^^H000000:bjiP2T]]h0[;[V0:BfgP2/^^H2000000X0[;kV100000002P2T
_^H0[<;^0:BniP2/`^h0Y;kV0:c2kP2T_^H0[<;^0:BniP2/`^h3000000080:c6kP2T`^h000000:Bn
iP2/`^h0Y;kV0:c2kP2T_^H3000000070:C2kP2Ta^h0Y<;^0:C6kP2T`^h0Y<K^0:C2kP040000000:
0:C6kP2T`^h0Y<K^0:C2kP2Ta^h0Y<;^0:C6kP2T`^h0Y<K^0:C2kP@0000000<0Y<K^0:C2kP000000
403oool3000002<0oooo000F0?ooo`800000103oool00`000000oooo0?ooo`060?ooo`800000;@3o
ool3000000090:BjiP2/^^H000000:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV00<0000000/0Y:[N0:b^
gP2TZ]h0[:kN0:BZgP2/[]h0Y:[N0:b^gP000000[;[V0:BfiP030000000;0:BZe@2/Z]h0Y:[E0:bZ
gP2TZ]D0[:[N0:BZe@2/Z]h0Y:[E0:bZgP2TZ]D00`0000002`2T[]h0[:kN0:B^gP2/[]h0Y:kN0:b^
gP2T[]h0[:kN0:B^gP2/[]h0Y:kN00<000000P2//]h020000000[;KV0:bbgP2/]^H0[;;N0:bfiP2/
/]h0[;KV0`0000070:bjiP030000002/^^H0[;KV0080000000X0[;k^0:bniP2/_^h0[;kV0:bnkP2/
_^H0[;k^0:bniP2/_^h0[;kV100000002@2/`^h0[;kV0:c2kP2/_^H0[<;^0:bniP2/`^h0[;kV0:c2
kP03000000080:c6kP2Ta^h0[<K^0:C6kP2/a^h000000:c2kP2T`^h3000000080:C2kP2/a^h0Y<;^
0:c6kP2T`^h0[<K^0:C2kP2/a^h30000000:0:c6mP2Ta^h000000:C2kP2/a_H0Y<;^0:c6mP2T`^h0
[<Kf0:C2kP<0000000P0Y<K^0:c6mP2Ta^h0[<Kf0:C6kP2/a_H0Y<K^00000140oooo00<000000?oo
o`3oool08@3oool001`0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0V0?ooo`@0
000000H0Y;KV0:BjiP2T]^H0Y;[V0:BfiP2T^^H20000000?0:BniP2T_^h000000000002TZ]D0Y:kN
0:BZe@2T[]h0Y:[E0:B^gP2TZ]D0Y:kN0:BZe@2T[]h0Y:[E00<0000000`0Y:[E0:BZgP2TZ]D0Y:[N
0:BZe@2TZ]h0Y:[E0:BZgP2TZ]D0Y:[N0:BZe@2TZ]h20000000<0:BUe@2/Z]D000000:b^gP2T[]D0
[:kN0:B^e@2/[]h0Y:kE0:b^gP2T[]D0[:kN0P0000002`2//]D0[;;N0:bbe@2//]h0[;;E0000002T
/]h0[;;N0:BbgP2//]h0Y;;N0080000000X0[;[V0:bfgP2/^^H0[;KN0:bjiP2/]]h0[;[V0:bfgP2/
^^H0[;KN0`0000090:bniP<0000000X0Y<K^0:c6kP2Ta^h0[<K^0000002/`^h0Y;kV0:c2kP2T_^H0
[<;^0`000000202/a^h0Y<;^0:c6kP2T`^h0[<K^0:C2kP2/a^h0Y<;^0`0000002@2T`^h0Y<K^0:C2
kP2Ta^h0Y<;^0:C6kP2T`^h0Y<K^0:C2kP02000000090:C6kP2T`^h0Y<K^0000002Ta^h0Y<;^0000
002T`^h0Y<K^00@0000000X0Y<;^0:C6kP2T`^h0Y<K^0:C2kP2Ta^h0Y<;^0:C6kP2T`^h0Y<K^0P00
000A0?ooo`030000003oool0oooo0200oooo000K0?ooo`<000002P3oool00`000000oooo0?ooo`0R
0?ooo`<0000000/0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP030000
000;0:b^gP2TZ]h0[:kN0:BZgP2/[]h0Y:[N0:b^gP2TZ]h0[:kN0:BZgP2/[]h00P0000003@2TWLd0
[:7E0000002/Z]h0Y:[E0:bZgP2TZ]D0[:[N0:BZe@2/Z]h0Y:[E0:bZgP2TZ]D00P0000040:bZe@07
0000002T[]h0[:kN0:B^gP2/[]h0Y:kN0:b^gP03000000P0[;;N00<000000:bbgP2/]^H00P00000:
0:bjiP80000000X0[<;^0000002/_^H0[;k^0:bniP2/_^h0[;kV0:bnkP2/_^H0[;k^0P0000070:c6
kP030000002/_^H0[<;^00<0000000P0[<K^0:C6kP2/a^h0Y<K^0:c6kP2Ta^h0[<K^0:C6kP<00000
00T0Y<K^0:c:mP000000[<K^0:C2kP2/a^h0Y<;^0:c6kP2T`^h00`000000202Ta^h0[<Kf0:C6kP2/
a_H0Y<K^0:c6mP000000[<Kf0`0000002P2/a_H0Y<K^0:c6mP2Ta^h0[<Kf0:C6kP2/a_H0Y<K^0:c6
mP2Ta^h4000000030:c6kP2T`^h000000140oooo00<000000?ooo`3oool07`3oool001/0oooo00<0
00000?ooo`3oool02`3oool00`000000oooo0?ooo`0N0?ooo`<0000000/0Y;[V0:BfiP2T^^H0Y;KV
0:BjiP2T]^H0Y;[V0:BfiP2T^^H0Y;KV0:BjiP030000000<0:BMc@2LVLd000000:BZe@2T[]h0Y:[E
0:B^gP2TZ]D0Y:kN0:BZe@2T[]h0Y:[E0P000000302TXLd0Y9g=0:BQc@2TWLd000000:BZe@2TZ]h0
Y:[E0:BZgP2TZ]D0Y:[N0:BZe@<0000000X0Y:GE0:bZe@2TYMD0[:[E0:BUe@2/Z]D0Y:GE0000002T
[]D0[:kN0`0000002`2//]h0[;;E0:bbgP2//]D0[;;N0:bbe@2//]h0[;;E0:bbgP2//]D0[;;N00<0
000000X0[;[V0:bfgP2/^^H0[;KN0:bjiP2/]]h0[;[V0:bfgP2/^^H0[;KN0P0000001@2/a^h0[<;V
0:c6kP2/`^H0000000@0[;kV0`0000002@2/a^h0Y<K^0:c6kP2Ta^h0[<K^0:C6kP2/a^h0Y<K^0:c6
kP03000000090:c6kP2T`^h0[<K^0:C2kP2/a^h0Y<;^0:c6kP2T`^h0[<K^0080000000T0Y<K^0:C:
mP2Ta^h0Y<[f0:C6kP2Tb_H000000:C6kP2T`^h00`000000202T`^h0Y<K^0:C2kP2Ta^h0Y<;^0:C6
kP2T`^h0Y<K^0`0000002@2Ta^h0Y<;^0000002T`^h0Y<K^0:C2kP2Ta^h0Y<;^0:C6kP04000000H0
Y<;^00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0O0?ooo`006`3oool2000000d0oooo
00<000000?ooo`3oool06`3oool20000000<0:B^gP2TZ]h000000:BjiP2/^^H0Y;[V0:bjiP2T^^H0
[;[V0:BjiP2/^^H0Y;[V0P0000003@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0000002/[]h0Y:[N0:b^gP2T
Z]h0[:kN0:BZgP2/[]h00P000000302TWLd0[:7E0:BMc@2/XMD0Y9g=0:bQe@2TWLd000000:BZe@2/
Z]h0Y:[E0:bZgP8000002`2/Z]D2000000`0[;;N0P0000000`2d_^H0[;kV000000070:bjiP<00000
00P0[<;^0:c6kP2/`^h0[<K^0:c2kP2/a^h000000:bnkP<000002@2/a^h3000000090:c>mP2Tb^h0
00000:C6kP2/a^h0Y<K^0:c6kP2Ta^h0[<K^00<0000000P0[<[f0:C6kP2/b_H0Y<K^0:c:mP2Ta^h0
[<[f0:C6kP<0000000P0Y<K^0:c6mP2Ta^h0[<Kf0:C6kP2/a_H0Y<K^0:c6mP<0000000P0[<Kf0:C2
kP000000Y<;^0:c6mP000000[<Kf0:C6kP@0000000X0[<K^0:C2kP2/a^h0Y<;^0:c6kP2T`^h0[<K^
0:C2kP2/a^h0Y<;^0P00000@0?ooo`800000803oool001/0oooo00<000000?ooo`3oool03@3oool0
0`000000oooo0?ooo`0G0?ooo`<0000000/0Y:[E0:BZgP2TZ]D0Y:[N0000002T^^H0Y;KV0:BjiP2T
]^H0Y;[V0:BfiP030000000=09bIc@2TWLd0W9W=0:BMc@2LVLd0Y9g=09bIc@2TWLd000000:B^gP2T
Z]D0Y:kN0:BZe@020000000<0:BQc@2TWLd0Y:7=0:BMc@2TXLd0Y9g=0:BQc@2TWLd0Y:7=0000002T
Z]h0Y:[E0P0000002`2/Z]D0Y:GE0:bZe@2TYMD0[:[E0:BUe@2/Z]D0Y:GE0:bZe@2TYMD0[:[E0080
000000`0[;;E0000002//]D0[;;N0:bbe@2//]h0[;;E0:bbgP2//]D0[;;N0:bbe@2//]h2000000@0
[;kV00H000000:bjiP2/]]h0[;[V0:bfgP2/^^H2000000080:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^
0:c2iP2/a^h4000000090:C6kP2/a^h0Y<K^0:c6kP2Ta^h0[<K^0:C6kP2/a^h0Y<K^0080000000T0
[<[f0:C:kP2/b_H0Y<[^0:c:mP2Tb^h000000:C2kP2/a^h00`000000202Tb_H0Y<K^0:C:mP2Ta^h0
Y<[f0:C6kP2Tb_H0Y<K^0`000000202Ta^h0Y<Kf0000002Ta^h0Y<;^0:C6kP2T`^h0Y<K^0`000000
202Ta^h0Y<;^0:C6kP2T`^h0Y<K^0:C2kP000000Y<;^0`00000;0:C2kP<0000000<0Y;kV0:C2kP00
00004@3oool00`000000oooo0?ooo`0M0?ooo`006`3oool00`000000oooo0?ooo`0>0?ooo`030000
003oool0oooo01<0oooo0`0000002`2TZ]h0Y:kN0:BZgP2T[]h0Y:[N0:B^gP2TZ]h0Y:kN0000002/
^^H0Y;[V00<0000000h0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@00
0000Y:[N0:b^gP80000000/0Y9g=0:bQe@2TWLd0[:7E0:BMc@2/XMD0Y9g=0:bQe@2TWLd0[:7E0:BM
c@03000000/0[:[E0P000000102d]]h0[;;N0;BfgP0000070:bbgP<0000000P0];kV0:bniP2d_^H0
[;kV0;BniP2/_^H0];kV00000080[;[V0P0000001`2/a^h0[<;^0:c6kP2/`^h0[<K^0:c2kP2/a^h0
100000000`2/c_H0[=;f000000060:c6kP<0000000P0Y<[^0:c>mP2Tb^h0[<kf0:C:kP2/c_H0Y<[^
0:c>mP<0000000P0[<[f0:C6kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP<0000000P0Y<K^0:C:mP2T
a^h0Y<[f0:C6kP000000Y<K^0:c6mP<0000000P0[<Kf0:C2kP2/a_H0Y<;^0:c6mP2T`^h0[<Kf0:C2
kP<0000000/0Y<;^0000002T`^h0[<K^0:C2kP2/a^h0Y<;^0:c6kP2T`^h0[<K^0:C2kP0300000006
0:BnkP2/`^h0Y;k^0:c2kP2T_^h000004@3oool00`000000oooo0?ooo`0M0?ooo`006`3oool20000
0100oooo00<000000?ooo`3oool0403oool20000000;0:BZe@2TZ]h0Y:[E0:BZgP2TZ]D0Y:[N0:BZ
e@2TZ]h0Y:[E0:BZgP2TZ]D00`0000003P2LVLd0Y9g=09bIc@2TWLd0W9W=0:BMc@2LVLd0Y9g=09bI
c@2TWLd0W9W=0:BMc@2LVLd0Y9g=0`0000002`2TXLd0Y9g=0:BQc@2TWLd0Y:7=0:BMc@2TXLd0Y9g=
0:BQc@2TWLd0Y:7=0080000000`0[:G=0:bUe@000000[:[E0:BUe@2/Z]D0Y:GE0:bZe@2TYMD0[:[E
0:BUe@2/Z]D20000000;0:bbe@2/]]h0[;;E0:bfgP2//]D0[;KN0000002//]h0[;;E0:bbgP2//]D0
0P00000:0:bniP<0000000H0[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h4000000090:C>kP2/c_H0Y<k^
0:c>mP2Tc^h0[<kf0000002/a^h0Y<K^00<0000000L0Y<[^0:c:mP2Tb^h0[<[f0:C:kP2/b_H0Y<[^
00@0000000P0Y<kf0:C:kP000000Y<K^0:C:mP2Ta^h0Y<[f0:C6kP<0000000P0Y<K^0:C6mP2Ta^h0
Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP<0000000T0Y<K^0:C2kP2Ta^h0Y<;^0:C6kP2T`^h0Y<K^0:C2kP2T
a^h00P0000001P2T_^h0Y<;^0000002T`^h0Y;k^000000D0Y<;^0`000000202T_^H0Y<;^0:BniP2T
`^h0Y;kV0:C2kP2T_^H0Y<;^0P00000A0?ooo`030000003oool0oooo01`0oooo000J0?ooo`800000
4P3oool00`000000oooo0?ooo`0<0?ooo`<0000000/0Y:kN0:BZgP2T[]h0Y:[N0:B^gP2TZ]h0Y:kN
0:BZgP2T[]h0Y:[N0:B^gP020000000?09b9_@000000Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=
0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0080000000`0[9W50:BEa@000000Y9g=0:bQe@2TWLd0[:7E
0:BMc@2/XMD0Y9g=0:bQe@2TWLd2000000050:bUe@2/YLd0[:GE0:bUc@0000001`2/Z]D200000009
0;BfgP2//]h0];KN0:bbgP2d]]h0[;;N0;BfgP2//]h000000080[;;N0P0000002@2/_^H0];kV0:bn
iP2d_^H0[;kV0;BniP2/_^H0];kV0:bniP03000000060:c>kP2dd_H000000:c6kP2/`^h0[<K^1@00
0000202/d_H0[<kf0:cBmP2/c_H0[=;f0:c>mP2/d_H0[<kf0`0000001`2Tb^h0[<kf0:C:kP2/c_H0
Y<[^0:c>mP2Tb^h00`0000002@2Tb_H0Y<kf0:C:mP2Tc_H0Y<[f0:C>mP000000[<[f0:C6kP030000
00080:C6kP2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP2Tb_H300000080Y<;^00L000000:C2kP2/a_H0
Y<;^0:c6mP2T`^h0[<Kf00<0000000X0[<;^0:C2kP2/`^h0Y<;^0:c2kP000000[<;^0000002/a^h0
Y<;^0`000000202T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0`0000000`2/_^h0Y;kV0000
000A0?ooo`030000003oool0oooo01/0oooo00080?ooo`800000103oool2000000@0oooo0P000004
0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3oool02@3oool20000000<09bAa@2LSLD0
00000:BZe@2TZ]h0Y:[E0:BZgP2TZ]D0Y:[N0:BZe@2TZ]h0Y:[E0P0000003`2TRKd0W8Vm0:B9_@2L
RKd0000009bIc@2TWLd0W9W=0:BMc@2LVLd0Y9g=09bIc@2TWLd0W9W=0:BMc@02000000@0Y9G500P0
00000:BQc@2TWLd0Y:7=0:BMc@2TXLd0Y9g=0:BQc@80000000`0[:G=0:bUe@2/YLd0[:GE0:bUc@2/
YMD0[:G=0000002TYMD0[:[E0:BUe@2/Z]D20000000:0:bbe@2/]]h0[;;E0:bfgP2//]D0[;KN0:bb
e@2/]]h0[;;E0:bfgP<000002@2/_^H2000000050:c>kP2/c_H0[<k^0:c>mP2/c^h01P0000001`2/
c_H0Y<k^0:c>mP2Tc^h0[<kf0:C>kP2/c_H0100000020:CBmP80000000<0Y<[^0:c:mP0000000`00
00002@2Tb^h0Y<kf0:C:kP2Tc_H0Y<[^0:C>mP2Tb^h0Y<kf0:C:kP03000000090:C6kP2Ta_H0Y<K^
0:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^008000001P2T`^h00`000000Y<;^0:C6kP03000000090:C2
kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^00@0000000T0Y;kV0:C2kP2T_^H0Y<;^0:Bn
iP2T`^h0Y;kV0:C2kP2T_^H00P0000001`2T_^h0Y;[V0:BnkP2T^^H0Y;k^0:BjiP0000003`3oool2
000000P0oooo0P00000C0?ooo`001`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0
oooo000000<0oooo0P00000D0?ooo`030000003oool0oooo00D0oooo0`000000302LSLD0Y97509b=
a@2TTLD000000:B^gP2TZ]h0Y:kN0:BZgP2T[]h0Y:[N0:B^gP80000000h0W8Vm0:B=a@2LRKd0Y8g5
09b9_@2TSLD000000:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@<0000000`0[9W50:BEa@2/VLD0
Y9G50:bIa@2TULD0[9W50000002/XMD0Y9g=0:bQe@2TWLd20000000:0:bUe@2/YLd0[:GE0:bUc@2/
YMD0[:G=0:bUe@2/YLd0[:GE00000080[:[E0P0000002P2d]]h0[;;N0;BfgP2//]h0];KN0:bbgP2d
]]h0[;;N0;BfgP2//]h2000000090;C>kP2db^h000000:bniP2d_^H0[;kV0;BniP2/_^H0];kV00<0
000000H0]=;f0:c>kP2dd_H0[<k^0;CBmP2/c^h4000000070:cBmP2/c_H0[=;f0:c>mP2/d_H0[<kf
0:cBmP03000000070:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`030000000:0:C>mP2Tb_H0Y<kf
0:C:mP2Tc_H0Y<[f0:C>mP2Tb_H0Y<kf0:C:mP<0000000T0Y<K^0:C6mP000000Y<[f0:C6kP2Tb_H0
Y<K^0:C:mP2Ta^h00`0000080:C2kP<0000000T0[<;^0:C2kP2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2
kP2/`^h01@000000202/_^h000000:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0`000000202T_^H0[;k^
0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0P00000@0?ooo`030000003oool0oooo00@0oooo00@00000
0?ooo`3oool000004P3oool000L0oooo00@000000?ooo`3oool000002`3oool01@000000oooo0?oo
o`3oool0000001H0oooo00@000000?ooo`3oool0oooo0`0000003@2LSLD0W97509b=a@2LTLD0W8g5
09bAa@2LSLD0W9750000002TZ]h0Y:[E0:BZgP2TZ]D00P0000003P2TRKd0W8Vm0:B9_@2LRKd0Y8Vm
09b9_@2TRKd0W8Vm0:B9_@000000Y9g=09bIc@2TWLd0W9W=0P00000:0:BEa@030000002TWLd0Y:7=
0080000000`0[:G=0:bUe@2/YLd0[:GE0:bUc@2/YMD0[:G=0:bUe@2/YLd0[:GE0:bUc@2/YMD20000
000:0:bbe@2/]]h0[;;E0:bfgP2//]D0[;KN0:bbe@2/]]h0[;;E0:bfgP80000000H0[<[V0;C:kP2/
b^H0]<[^0:c:iP0000020:bniP<0000000H0[<kf0:c>kP2/c_H0[<k^0:c>mP2/c^h4000000060:CJ
o`2Tf_H000000000002/c_H0Y<k^100000070:CBmP<0000000T0Y<kf09c>mP000000Y<[^0:C>mP2T
b^h0Y<kf0:C:kP2Tc_H0100000002@2La^h0Y<Kf09c6kP2Ta_H0W<K^0:C6mP000000Y<Kf0:C6kP03
000000P0Y<;^0`0000002@2T_^h0Y;[V0000002T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP030000000:
0:BnkP2T^^H0Y;k^0:BjiP000000Y;[V0:BnkP000000Y<;^0:BniP<0000000P0Y;[V0:BnkP2T^^H0
Y;k^0:BjiP2T_^h0Y;[V0:BnkP<0000000<0[;[V0000003oool03`3oool00`000000oooo0?ooo`06
0?ooo`030000003oool0oooo0140oooo00000`3oool000000000000200000080oooo00@000000?oo
o`3oool000002@3oool2000000@0oooo00<000000?ooo`3oool05@3oool30000000>09b=a@2TTLD0
W8g50:BAa@2LSLD0Y97509b=a@2TTLD0W8g50:BAa@2LSLD000000:BZgP2T[]h20000000>09b9_@2T
SLD0W8Vm0:B=a@2LRKd0Y8g509b9_@2TSLD0W8Vm0:B=a@2LRKd000000:BIc@2TWLd20000000=0:BE
a@2/VLD0Y9G50:bIa@2TULD0[9W50:BEa@2/VLD0Y9G50:bIa@2TULD0[9W50:BEa@020000000<0:bU
e@2/YLd0[:GE0:bUc@2/YMD0[:G=0:bUe@2/YLd0[:GE0:bUc@2/YMD0[:G=0P0000002@2d_^H00000
0;BfgP2//]h0];KN0:bbgP2d]]h0[;;N0;BfgP03000000070;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^
0;C>kP03000000060;CBmP2/c^h0]=;f0:c>kP2dd_H0[<k^1@0000001@2Tf_l0Y=ko0:CJo`2Tg_l0
Y=[o00<0000000T0Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H00P0000002@2Tc_l0
W<kf0:C>o`2Lc_H0Y<ko09c>mP000000Y<[f0:C>mP030000000:0:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0
Y<K^0:C6mP2Ta^h0Y<Kf0:C6kP<000002@2T`^h2000000080:bnkP2T_^H0[;k^0:BniP2/_^h00000
0:c2kP2T`^h40000000:0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H000000:BniP<00000
00P0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP<0000000H0[;k^0:BjiP2/_^h0Y;[V0:bn
kP00000@0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0403oool000L0oooo00@0
00000?ooo`3oool000002@3oool00`000000oooo0?ooo`030?ooo`800000503oool30000000>09bA
a@2LSLD0W97509b=a@2LTLD0W8g509bAa@2LSLD0W97509b=a@2LTLD0W8g509bAa@2LSLD20000000>
0:B9_@2LRKd0Y8Vm09b9_@2TRKd0W8Vm0:B9_@2LRKd0Y8Vm09b9_@2TRKd0W8Vm0:B9_@2LRKd20000
00d0Y9G50`0000002`2/YMD0[:G=0:bUe@2/YLd0[:GE0:bUc@2/YMD0[:G=0:bUe@2/YLd0[:GE0080
000000T0];[N0;BngP2d^]h000000:bbe@2/]]h0[;;E0:bfgP2//]D00P000000202db^h0[<[V0;C:
kP2/b^H0]<[^0:c:iP2db^h0[<[V0P0000001P2/h_l0Y>;f0000002/c^h0[<kf0:c>kPD0000000H0
Y=[f0:CJo`2Tf_H0Y=[o0:CJmP2Tf_l200000080W=[o00<000000:CBmP2Td_H0102Td_H300000008
09c>mP2Tc_H0W<kf0:C>mP2Lc_H0Y<kf09c>mP2Tc_H3000000090:C6mP2La^h0Y<Kf09c6kP2Ta_H0
W<K^0:C6mP2La^h0Y<Kf00@0000000<0Y;kV0:BnkP0000001P2T`^h3000000070:BjiP2T_^h0Y;[V
0:BnkP2T^^H0Y;k^0:BjiP030000000;0:BjiP2T_^h0Y;[V0:BnkP2T^^H0Y;k^0:BjiP2T_^h0Y;[V
0:BnkP2T^^H00`000000202T^^H0[;[V0000002T_^h0Y;[V0:BnkP2T^^H0Y;k^0`0000002@2/^^H0
Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0000000A0?ooo`030000003oool0oooo0080oooo00@0
00000?ooo`3oool000004P3oool000P0oooo0P00000:0?ooo`<000000`3oool00`000000oooo0?oo
o`0A0?ooo`80000000l0U7F/0000002TTLD0W8g50:BAa@2LSLD0Y97509b=a@2TTLD0W8g50:BAa@2L
SLD0Y97509b=a@2TTLD00`0000003@2TSLD0W8Vm0:B=a@2LRKd0Y8g509b9_@2TSLD0W8Vm0:B=a@2L
RKd0Y8g509b9_@2TSLD00`000000302/VLD0Y9G50:bIa@2TULD0[9W50:BEa@2/VLD0Y9G50:bIa@2T
ULD0[9W50:BEa@80000000`0]:[E0;BZc@2dZ]D000000:bUe@2/YLd0[:GE0:bUc@2/YMD0[:G=0:bU
e@2/YLd2000000090;BniP2d_]h0];kV0;BngP2d_^H0];kN0000002//]h0];KN0080000000P0]<[^
0;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^0;C>kP80000000D0Y>;o0:cRo`2Th_l0[>;o0:CRo`050000
00060:CJo`2Tg_l0Y=[o0:CNo`2Tf_l0Y=ko0`00000509cJo`030000002Te_l0Y=;f00<0000000L0
W<kf0:C>o`2Lc_H0Y<ko09c>mP2Tc_l0W<kf00@0000000T0Y<Kf0:C2kP000000Y<K^0:C6mP2Ta^h0
Y<Kf0:C6kP2Ta_H00`0000070:BnkP030000002T`^h0Y<;^00<0000000L0Y;kV0:bnkP2T_^H0[;k^
0:BniP2/_^h0Y;kV00<0000000`0Y;[V0:bjkP000000[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^
0:BniP2/_^h2000000080:BjiP2/_^h0Y;[V0000002T^^H000000:BniP2/_^h30000000;0:bnkP2T
^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h00P00000@0?ooo`030000003oool0
oooo00<0oooo0P00000C0?ooo`006@3oool00`000000oooo0?ooo`0@0?ooo`80000000l0U76T09Aa
[02DLJ@0U76/0000002LTLD0W8g509bAa@2LSLD0W97509b=a@2LTLD0W8g509bAa@2LSLD00P000000
3P2LPK@0W7f/09b1]0000000Y8Vm09b9_@2TRKd0W8Vm0:B9_@2LRKd0Y8Vm09b9_@2TRKd0W8Vm0P00
0000102/TKd0Y96m0:bA_@0000090:BEa@80000000`0[:G=0;BZc@2/YLd0]:[=0:bUc@000000[:G=
0:bUe@2/YLd0[:GE0:bUc@2/YMD2000000080;BjgP2d_]h0];[N0;BngP2d^]h0];kN0;BjgP2d_]h3
000000070;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V0;C:kP03000000040:cRo`2Th_H0[>;o0:CRmPH0
000000D0Y=[o0:CJmP2Tf_l0Y=[f0:CJo`03000000L0W=[o100000001`2Lc_H0Y<kf09c>mP2Tc_H0
W<kf0:C>mP2Lc_H00`0000002@2L`^h0Y<;f09c2kP2T`_H0W<;^0:C2mP000000Y<Kf09c6kP040000
000:0:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^0:BniP<0000000P0Y;[V0:BnkP2T
^^H0Y;k^0:BjiP2T_^h0Y;[V0:BnkP8000001@2T^^H01`000000Y;[V0:BnkP2T^^H0Y;k^0:BjiP2T
_^h00`0000001`2/^^H0Y;[V0:bjiP2T^^H0[;[V0000002/^^H00`0000002`2/^^H0Y;[V0:bjiP2T
^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V00<0000000<0[;kV0:BjiP0000003P3oool00`00
0000oooo0000000I0?ooo`006@3oool2000000h0oooo0`0000003`2DMJ`0U76/09Ae[02DLJ`0U7F/
09Aa[0000000W8g50:BAa@2LSLD0Y97509b=a@2TTLD0W8g50:BAa@020000000[09am[02TPK@0W7f/
0:B1]02LOJ`0000009b9_@2TSLD0W8Vm0:B=a@2LRKd0Y8g509b9_@2TSLD0W8Vm0000002/TKd0[9G5
0:bA_@2/ULD0[96m0000002TULD0[9W50:BEa@2/VLD0Y9G50:bIa@2TULD0[9W50000002dZ]D0]:[=
0;BZe@2dZ/d0]:[E0;BZc@2dZ]D0]:[=0000002/YLd0[:GE0:bUc@02000000080;BniP2d_]h0];kV
0;BngP2d_^H0];kN0;BniP2d_]h2000000080;CVo`2/h_H000000;C:kP2dc^h0]<[^0;C>kP2db^h2
000000040:cRo`2Th_l0[>;o0:CRo`H0000000H0U>Ko08_Ro`00000000000:CNo`2Tf_l2000000H0
W=[o100000000`2Lc_l0W<kf09c>o`020000000309c>mP2Tc_l0000000<0000000T0Y<;^0:C6mP2T
`^h0Y<Kf0:C2kP2Ta_H0Y<;^0:C6mP2T`^h00`00000:0:BnkP@0000000P0Y;KV0:BjiP000000[;k^
0:BniP2/_^h0Y;kV0:bnkP<0000000/0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP000000
Y;kV0:bnkP03000000070:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP030000000;0:bnkP000000
[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h00`0000001P2/_^h0Y;kV0:bnkP2T_^H0
[;k^00000140oooo00<000000?ooo`3oool05P3oool001T0oooo00<000000?ooo`3oool02`3oool2
0000000@09Aa[02DLJ@0U76/09AaY02DLJ`0U76T09Aa[02DLJ@0U76/09AaY0000000W8g509bAa@2L
SLD0W97509b=a@80000000l0W86d09am[02LPK@0W7f/09b1]02LOJ`0W86d0000002TRKd0W8Vm0:B9
_@2LRKd0Y8Vm09b9_@2TRKd00P000000202TTKd0[96m0:BA_@2/TKd0Y96m0:bA_@2TTKd000001@2T
ULD20000000;0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0000002/YMD00P000000
202d^]h0];kN0;BjgP2d_]h0];[N0;BngP2d^]h0];kN0P000000202/h_H0]>;f0:cRmP2dh_H0[>;f
0000002/b^H0]<[^0P0000000`2Th_H0[>;o0:CRmP050000000608_RmP2Di_l0Rn;f09CVo`2;h_H0
U>Ko1000000409cJo`@0000000P0W<[f09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>o`<0000000X0
Y<;f09c2kP2T`_H0W<;^0:C2mP2L`^h0Y<;f09c2kP2T`_H0W<;^0`0000002P2L]^H0Y;[V0000002T
_^h0Y;kV0:BnkP2T_^H0Y;k^0:BniP2T_^h3000000090:BjiP2T]^H0Y;[V0:BfiP2T^^H0Y;KV0000
002T^^H0Y;k^00<000002`2T^^H3000000080:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H2
0000000;0:bniP2T^^H0[;kV0000002/_^H000000:bjiP2T^^H0[;[V0:BjiP2/^^H00`000000202/
_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0P00000@0?ooo`030000003oool0oooo01H0oooo
000I0?ooo`030000003oool0oooo00P0oooo0`000000402DLJ`0U7F/09Aa[02DMJ`0U76/09Ae[02D
LJ`0U7F/09Aa[02DMJ`0U76/09Ae[0000000Y97509b=a@2TTLD20000000N09am[02TPK@0W7f/0:B1
]02LOJ`0Y86d09am[02TPK@0W7f/0:B1]0000000Y8g509b9_@2TSLD0W8Vm0:B=a@000000[9G50:bA
_@2/ULD0[96m0:bEa@2/TKd0[9G50:bA_@2/ULD000000:bIa@2TULD0[9W50P0000005@2dZ/d0]:[E
0;BZc@2dZ]D0]:[=0;BZe@2dZ/d0]:[E0;BZc@2dZ]D0]:[=0;BZe@000000];kV0;BngP2d_^H0];kN
0;BniP2d_]h0];kV0;BngP02000000070;CVo`2/h_H0]>Ko0:cRmP2di_l0[>;f0;CVo`0300000003
0:cRo`2Th_l0[>;o00@0000000D0Rn;o09CVo`2;h_l0U>Ko08_Ro`040000000409CNo`2;f_l0U=ko
08_Jo`@0000000P0W<ko09c>mP2Lc_l0W<kf09c>o`2Lc_H0W<ko09c>mP@0000000T0Y;k^09bjkP00
0000Y<;^0:C6mP2T`^h0Y<Kf0:C2kP2Ta_H0100000001`2T]^H0Y;[^0:BfiP2T^^h0Y;KV0:BjkP00
00000P2T_^h4000000090:BjiP2T]^H0Y;[V0:BfiP2T^^H0Y;KV0:BjiP2T]^H0Y;[V00<0000000X0
[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V10000000202/^^H0Y;[V0000002T
^^H0[;k^0:BjiP2/_^h0Y;[V0`0000002P2T^^H0[;k^0:BjiP2/_^h0Y;[V0000002T^^H000000:Bj
iP2/_^h3000000080:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H3000000030:BnkP000000
oooo0100oooo00<000000?ooo`3oool05@3oool001T0oooo0P0000060?ooo`<000000100RefD0000
002DLJ@0U76/09AaY02DLJ`0U76T09Aa[02DLJ@0U76/09AaY02DLJ`0U76T09Aa[02DLJ@0U76/0`00
0000402LPK@0W7f/09b1]02LOJ`0W86d09am[02LPK@0W7f/09b1]02LOJ`0W86d09am[0000000W8Vm
0:B9_@2LRKd20000000;0:bA_@2TTKd0[96m0:BA_@2/TKd0Y96m0:bA_@2TTKd0[96m0:BA_@2/TKd0
0`000000302dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@800000
00L0_MK^0000002d_]h0];[N0;BngP2d^]h0];kN0080000000L0[>;f0;CRmP2/h_H0]>;f0:cRmP2d
h_H0[>;f0080000000<0PnK^07_ViP23i^h00`000000102Di_l0Rn;f09CVo`2;h_H5000000@0Rm[o
100000002@2D`_H0W<Kf0000002Lc_l0W<[f09c>o`2Lb_H0W<ko09c:mP030000000:09bjiP2T^^h0
W;[V0:BjkP2L^^H0Y;[^09bjiP000000W<;^0:C2mP<0000000X0Y;[V09bfiP2T^^H0W;KV0:BjiP2L
]^H0Y;[V09bfiP2T^^H0W;KV0`0000002@2T]^H0Y;[V0:BfiP2T^^H0Y;KV0:BjiP2T]^H0Y;[V0:Bf
iP04000000030:BfiP2T]]h0000000L0Y;[V0`0000002@2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP00
0000[;[V0:BjiP03000000090:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V00@00000
00T0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H00P0000001P2T_^H0Y<;^0:BniP2T
`^h0Y;kV00000140oooo00<000000?ooo`3oool0503oool001T0oooo00@000000?ooo`3oool0oooo
100000008`2;GI@0RefL08]MU02;GI`0000009Ae[02DLJ`0U7F/09Aa[02DMJ`0U76/09Ae[02DLJ`0
U7F/09Aa[02DMJ`0U76/09Ae[02DLJ`0000009am[02TPK@0W7f/0:B1]02LOJ`0Y86d09am[02TPK@0
W7f/0:B1]02LOJ`0Y86d09am[02TPK@0W7f/00<0000001X0[96m0:bEa@2/TKd0[9G50:bA_@2/ULD0
[96m0:bEa@2/TKd0[9G50:bA_@2/ULD0[96m0000002dZ/d0]:[E0;BZc@2dZ]D0]:[=0;BZe@2dZ/d0
]:[E0;BZc@2dZ]D0]:[=0;BZe@80000000L0_MK^0;gJkP2me^h0_M[^0000002d_^H0];kN00800000
00H0]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H30000000307_VkP23i^h000000080000000<0H]KV05[F
gP000000100000001P2;f_l0U=ko08_Jo`2Dg_l0Rm[o09CNo`<000001P2La_H00`000000W<kf09c>
o`040000000:09bjkP2T_^h0W;[^0:BnkP2L^^h0Y;k^09bjkP2T_^h0W;[^0:BnkP<0000000T0Y;[^
0:BfiP2T^^h0Y;KV0:BjkP2T]^H0Y;[^0:BfiP2T^^h0100000002@2T/]h0[;;V0000002T^^H0Y;KV
0:BjiP2T]^H0Y;[V0:BfiP030000000:0:BfiP2/]^H0Y;KV0:bfiP2T]^H0[;KV0:BfiP000000Y;[V
0:bjkP@0000000T0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H00`0000002@2T^^H0
[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP05000000080:BniP000000Y;kV0:bnkP2T_^H0
[;k^0:BniP2/_^h3000000080:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP00000@0?ooo`800000
5P3oool001P0oooo00<000000?ooo`0000000P00000808]MU00>0000002DLJ@0U76/09AaY02DLJ`0
U76T09Aa[02DLJ@0U76/09AaY02DLJ`0U76T09Aa[02DLJ@20000000M09b1]02LOJ`0W86d09am[02L
PK@0W7f/09b1]02LOJ`0W86d09am[02LPK@0W7f/09b1]02LOJ`0W86d0000002/TKd0Y96m0:bA_@2T
TKd0[96m0:BA_@2/TKd0Y96m0:bA_@2TTKd0[96m0:BA_@2/TKd00P0000002`2m^]D000000:bUc@2d
Z/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=008000001P2me^h3000000050:cRmP2dh_H0[>;f
0;CRmP2/h_H00`0000000`1ki^H00000000000020000000305[FiP00000000000080000000<0Nm;o
08?Fo`1kd_l00P00000208_Jo`@0000000P0W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP@0
000000X0Y;[^09bjiP2T^^h0W;[V0:BjkP2L^^H0Y;[^09bjiP2T^^h0W;[V100000002@2T[]h0W:kN
0000002L]^H0Y;[V09bfiP2T^^H0W;KV0:BjiP03000000090:BbgP2T[]h0Y;;N0:B^gP2T/]h0Y:kN
0000002T]^H0Y;[V00@0000000X0Y;KN0:BfiP2T]]h0Y;KV0:BfgP2T]^H0Y;KN0:BfiP2T]]h0Y;KV
0`0000002P2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H3000000090:BjiP2/
_^H000000:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V00<0000000X0Y;kV0:bnkP2T_^H0[;k^0:BniP00
0000Y;kV0000002T^^H0[;kV0`0000002P2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2
kP2T_^H200000140oooo00<000000?ooo`3oool04`3oool001P0oooo0P0000005`2;GI`0RefD08]M
W02;GI@0RefL08]MU02;GI`0RefD08]MW02;GI@0RefL08]MU0000000U76/09Ae[02DLJ`0U7F/09Aa
[02DMJ`0U76/09Ae[02DLJ`0U7F/008000000100Y7VT0:Am[0000000Y86d09am[02TPK@0W7f/0:B1
]02LOJ`0Y86d09am[02TPK@0W7f/0:B1]02LOJ`0Y86d0P0000002`2/TKd0[9G50:bA_@2/ULD0[96m
0:bEa@2/TKd0[9G50:bA_@2/ULD0[96m0080000000/0_K[N0;fje@2m^]h000000;BZe@2dZ/d0]:[E
0;BZc@2dZ]D0]:[=0;BZe@02000000060;gFkP2mf^h0_MK^0;gJkP2me^h0_M[^0P0000001P1ki]D0
NnKN000000000000[>;f0;CVo`80000000<0NnK^0000000000001P0000001P23e_l0Nm;o08?Fo`1k
d_l0PmKo07_Bo`D00000202La_H40000000;09b^gP2T[^H0W:kN0000002L^^h0Y;k^09bjkP2T_^h0
W;[^0:BnkP2L^^h00`0000002@2T[]h0Y:kV0:B^gP2T[^H0Y:kN0:B^iP000000Y;[^0:BfiP040000
00090:bbiP2T/]h0[;;V0:BbgP2//^H0Y;;N0:bbiP2T/]h0[;;V00<0000000/0[;KV0:BfiP2/]^H0
Y;KV0:bfiP2T]^H0[;KV0:BfiP2/]^H0Y;KV0:bfiP030000000;0:bjiP000000[;[V0:BjiP2/^^H0
Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H00P000000202T_^H0[;k^0:BniP2/_^h0Y;kV0000002T^^H0
[;k^100000002P2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0000002/_^h3000000090:c2
kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^00@0000000<0Y<;^0:c6kP0000004@3oool0
0`000000oooo0?ooo`0B0?ooo`00603oool00`000000RefD08]MU00;08]MU00K0000002DLJ`0U76T
09Aa[02DLJ@0U76/09AaY02DLJ`0U76T09Aa[0000000Y7fT09aiY02TOJ@0W7VT0000002LOJ`0W86d
09am[02LPK@0W7f/09b1]02LOJ`0W86d09am[02LPK@0W7f/0080000000d0]9Vm0:bE_@000000Y96m
0:bA_@2TTKd0[96m0:BA_@2/TKd0Y96m0:bA_@2TTKd0000000H0_K[E00D000000;BZc@2/YLd0]:[=
0:bUc@02000000H0_MK^0P000000101ki]h0Ln;E07_VgP1ch]D;0000000407_Bo`23e_l0Nm;o08?F
o`H0000000@0U;;V09bbkP2D/^H0W;;^0P0000000`2D`_H0W<Kf09C2mP040000000;09b^iP2LZ]h0
W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h0000009bjiP2T^^h0100000002@2L[]h0Y:kN09b^gP2T[]h0
W:kN0:B^gP2L[]h0Y:kN09b^gP030000000;0:B^gP2T/]h0Y:kN0:BbgP2T[]h0Y;;N0:B^gP2T/]h0
Y:kN0:BbgP2T[]h00P000000302//]h0Y;;N0000002T]]h0Y;KV0:BfgP2T]^H0Y;KN0:BfiP2T]]h0
Y;KV0:BfgP80000000/0[;[V0:BfgP2/^^H0Y;KN0:bjiP000000[;[V0:BjiP2/^^H0Y;[V0:bjiP03
000000070:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP030000000;0:bnkP2T_^H0[;k^0:BniP2/
_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h00`0000002@2/`^h0Y<;^0000002T_^H0Y<;^0:BniP2T
`^h0Y;kV0:C2kP03000000H0Y<;^00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`0B0?oo
o`004@3oool2000000D0oooo03D0000008]MU02;GI`0RefD08]MW02;GI@0RefL08]MU02;GI`0RefD
08]MW02;GI@0RefL08]MU0000000U76/09Ae[02DLJ`0U7F/09Aa[02DMJ`0U76/09Ae[0000000Y7f/
0:AiY02TOJ`0Y7VT0:Am[0000000Y86d09am[02TPK@0W7f/0:B1]02LOJ`0Y86d09am[02TPK@0W7f/
0000002dUKd0]9Vm0;BE_@2dVKd000000:bEa@2/TKd0[9G50:bA_@2/ULD0[96m0:bEa@020000000:
0;fje@2m^]h0_K[E0;fjgP2m^]D0_K[N0;fje@000000]:[=0;BZe@80000000H0_MK^0;gJkP2me^h0
_M[^0;gFkP2mf^h20000000307_Ve@1ki]h0000000P0000000D0Pl;o07_2o`23`_l0Nl;o08?2o`05
0000000;09bbkP2D/^H0W;;^09BbiP2L/^h0U;;V09bbkP2D/^H0W;;^09BbiP2L/^h010000000302L
[]h0Y:kV09b^gP2T[^H0W:kN0:B^iP2L[]h0Y:kV09b^gP2T[^H0W:kN0:B^iP<0000000/0Y:kV0:B^
gP2T[^H0Y:kN0:B^iP2T[]h0Y:kV0:B^gP2T[^H0Y:kN0:B^iP020000000;0:BZe@2/Z]h000000:bb
iP2T/]h0[;;V0:BbgP2//^H0Y;;N0:bbiP2T/]h00`0000002`2T/]h0[;;V0:BbgP2//^H000000:bf
iP2T]^H0[;KV0:BfiP2/]^H0Y;KV00<0000000X0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0000
002T^^H0[;[V0`000000202/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0P000000302/`^h0
Y<;^0000002T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP80000000P0[<;^0:C2kP2/
`^h000000:c2kP000000[<;^0:BnkP@0000000T0[<K^0:C2kP2/a^h0Y<;^0:c6kP2T`^h0[<K^0:C2
kP2/a^h00P00000A0?ooo`030000003oool0oooo0140oooo000@0?ooo`040000003oool0oooo0000
00@0oooo00<0000008]MU02;GI@02`2;GI@01`000000U76/09AaY02DLJ`0U76T09Aa[02DLJ@00P00
00007P2TOJ@0W7VT0:AmY02LNJ@0Y7fT09aiY0000000W7f/09b1]02LOJ`0W86d09am[02LPK@0W7f/
09b1]02LOJ`000000:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0000002/TKd0Y96m0:bA_@2TTKd0[96m
000000X0_K[E0P0000060;gFkPP0000000P0Pl;o000000000000000008?2o`1k`_H0Pl;o07_2mP@0
000000d0W9WE09BEc@000000U;;V09bbkP2D/^H0W;;^09BbiP2L/^h0U;;V09bbkP2D/^H0W;;^00<0
000000d0W9gE09bIc@000000W:[N09b^iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h0W:kV00<0
000000/0Y:7E09bQc@000000W:kN0:B^gP2L[]h0Y:kN09b^gP2T[]h0W:kN0:B^gP030000000:0:BZ
gP2TZ]D0Y:[N0:BZe@000000Y:kN0:BbgP2T[]h0Y;;N0:B^gP<0000000/0Y;;N0:bbgP2T/]h0[;;N
0:BbgP2//]h0Y;;N0:bbgP000000Y;KV0:BfgP030000000:0:BfgP2/^^H0Y;KN0:bjiP2T]]h0[;[V
0:BfgP2/^^H0Y;KN0:bjiP<0000000P0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP<00000
00/0Y<;V0:c2kP2T`^H0[<;^0000002/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP03000000060:C2kP2/
`^h0Y<;^0:c2kP2T`^h0[<;^100000090:C2kP@000000P2Ta^h00`000000oooo0?ooo`0=0?ooo`03
0000003oool000000180oooo000@0?ooo`040000003oool0oooo000000@0oooo02D0000008]MU02;
GI`0RefD08]MW02;GI@0RefL08]MU02;GI`0RefD08]MW02;GI@0RefL08]MU0000000U76/09Ae[02D
LJ`0U7F/09Aa[0000000Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@0000009am[02T
PK@0W7f/0:B1]02LOJ`0Y86d0080000000/0]9Fm0;BI_@2dUKd0]9Vm0;BE_@2dVKd0]9Fm0000002/
TKd0[9G50:bA_@02000000090;fjgP2m^]D0_K[N0;fje@2m^]h0_K[E0;fjgP2m^]D0_K[N00800000
00H0W?;V000000000000_M[^0;gFkP2mf^h40000000408>IgP2;WNH0PiWN08^MiP<0000000<0Pl;o
07_2o`23`_l0100000003@2DUMD0W9WE09BEe@2LVMD0U9GE09bIe@000000W;;^09BbiP2L/^h0U;;V
09bbkP2D/^H010000000302TWMD0W9W=0:BMe@2LVLd0Y9gE0000002T[^H0W:kN0:B^iP2L[]h0Y:kV
09b^gP@000001@2TXMD01P000000Y:kV0:B^gP2T[^H0Y:kN0:B^iP<0000000/0[:[N0:BZe@2/Z]h0
Y:[E0:bZgP2TZ]D0[:[N0:BZe@000000Y;;N0:bbiP020000000;0:BbgP2//^H0Y;;N0:bbiP2T/]h0
[;;V0:BbgP2//^H0Y;;N0:bbiP2T/]h00`0000002P2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0
[;[V0:BjiP2/^^H3000000080:c2kP2/_^H000000:BniP2/_^h0Y;kV0:bnkP2T_^H30000000;0:C2
kP2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2kP2/`^h000000:bnkP2T_^H00`0000001P2T`^h0[<;^0:C2
kP2/`^h0Y<;^0:c2kP@0000000T0Y<K^0:c6mP000000[<K^0:C2kP2/a^h0Y<;^0:c6kP2T`^h00`00
00001P2Ta^h0[<Kf0:C6kP2/a_H0Y<K^0:c6mP800000403oool00`000000oooo0?ooo`0@0?ooo`00
403oool010000000oooo0?ooo`0000030?ooo`8000003@2;GI@08@000000U76/09AaY02DLJ`0U76T
0000002LNJ@0Y7fT09aiY02TOJ@0W7VT0:AmY02LNJ@0Y7fT09aiY02TOJ@0000009b1]02LOJ`0W86d
09am[02LPK@000000;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@03000000T0
_K[E0P0000001@2Lm^H0W?;N09cfiP2Ll]h0W?KV00<0000000H0RigN08>IgP2;WMh0PiWN08^MgP23
VMh20000000308^MgP000000000000<0000000h0W9WE09BEc@2LVMD0U9G=09bIe@2DULd0W9WE09BE
c@2LVMD0U9G=09bIe@000000W;;^09BbiP<0000000d0W9W=09bMe@2LVLd0W9gE09bIc@2LWMD0W9W=
09bMe@2LVLd0W9gE0000002L[^H0W:[N00<0000000d0W:7=0:BQe@2LXLd0Y:7E09bQc@2TXMD0W:7=
0:BQe@2LXLd0Y:7E0000002T[]h0W:kN0080000000/0Y:[N0:BZe@2TZ]h0Y:[E0:BZgP2TZ]D0Y:[N
0:BZe@2TZ]h0Y:[E0:BZgP030000000:0:bbgP2T/]h0[;;N0:BbgP2//]h0Y;;N0:bbgP2T/]h0[;;N
0:BbgP<0000000/0Y;[N0000002T]]h0[;[V0:BfgP2/^^H0Y;KN0:bjiP2T]]h0[;[V0:BfgP020000
00090:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0000002T^^H0[;kV0080000000/0Y<;V0:c2kP2T`^H0
[<;^0:C2iP2/`^h0Y<;V0:c2kP2T`^H0[<;^0:C2iP03000000060:C2kP2/`^h0Y<;^0:c2kP2T`^h0
[<;^0`0000001`2Ta^h0Y<;^0:C6kP2T`^h000000:C2kP0000000P2T`^h4000000H0Y<K^0`000000
0`2Ta^h0Y<;^0000000@0?ooo`030000003oool0oooo00l0oooo000@0?ooo`040000003oool0oooo
000000<0oooo0P000000402;GI@0RefL08]MU02;GI`0RefD08]MW02;GI@0RefL08]MU02;GI`0RefD
08]MW02;GI@0RefL0000002DMJ`20000000V0:AiY02TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0
Y7VT0:Am[02TNJ@0000009am[02TPK@0W7f/0:B1]0000000]9Vm0;BE_@2dVKd0]9Fm0;BI_@2dUKd0
]9Vm0;BE_@2dVKd0]9Fm0;BI_@000000_K[N0;fje@2m^]h0_K[E0;fjgP2m^]D0_K[N0;fje@2m^]h2
0000000509cbiP2Lm^H0W?;V09cfiP2Ll^H00P0000001P2;NKd0000008^MiP23VMh0RigV08>IgPH0
00000100W86m09B1_@2LPKd0000009bIe@2DUMD0W9WE09BEe@2LVMD0U9GE09bIe@2DUMD0W9WE09BE
e@2LVMD0U9GE0`0000003@2LVLd0Y9gE09bIc@2TWMD0W9W=0:BMe@2LVLd0Y9gE09bIc@2TWMD0W9W=
0:BMe@2LVLd00`00000=0:BQe@<0000000/0Y:[E0:bZgP2TZ]D0[:[N0:BZe@2/Z]h0Y:[E0:bZgP2T
Z]D0[:[N0:BZe@020000000;0:bbgP000000[;;V0:BbgP2//^H0Y;;N0:bbiP2T/]h0[;;V0:BbgP2/
/^H00P0000050:bjiP060000002T^^H0[;[V0:BjiP2/^^H0Y;[V0`000000202/_^H0[<;^0:bniP2/
`^h0[;kV0:c2kP2/_^H0[<;^0`0000002@2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2kP2/`^h0Y<;^0:c2
kP04000000050:C6kP2/a_H000000:c2kP2T`^h010000000202/a_H0Y<K^0:c6mP2Ta^h0[<Kf0:C6
kP2/a_H0Y<K^10000000202/a_H0Y<K^0:c6mP2Ta^h0[<Kf0:C6kP2/a_H0Y<K^0P0000001P2/a_H0
Y<K^0:c6mP2Ta^h0[<Kf00000140oooo00<000000?ooo`3oool03P3oool00140oooo0P0000040?oo
o`8000003P2;GI@20000000@0:AmY02LNJ@0Y7fT09aiY02TOJ@0W7VT0:AmY02LNJ@0Y7fT09aiY02T
OJ@0W7VT0:AmY0000000W86d09am[080000000h0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE
_@2dVKd0[9Fm0;BI_@000000cNk^000000H0_K[E0P000000102Lm^H0W?;N09cfiP2Ll]h300000003
08=i]02;NKd0PgVd00D000002@2DPKd02P000000U9G=09bIe@2DULd0W9WE09BEc@2LVMD0U9G=09bI
e@2DULd30000000>09b9_@2LSLD0000009bMe@2LVLd0W9gE09bIc@2LWMD0W9W=09bMe@2LVLd0W9gE
09bIc@2LWMD20000000>0:BEa@2TVLd000000:BQe@2LXLd0Y:7E09bQc@2TXMD0W:7=0:BQe@2LXLd0
Y:7E09bQc@2TXMD20000000;0:BQc@2/YMD000000:BZgP2TZ]D0Y:[N0:BZe@2TZ]h0Y:[E0:BZgP2T
Z]D00`0000002P2T[]D0[;;N0:B^e@2//]h000000:bbgP2T/]h0[;;N0:BbgP2//]h30000000:0:bj
iP2T^]h0[;[V0:BjgP2/^^H0Y;[N0:bjiP000000[;[V0:BfgP<0000000P0Y;kV0:bnkP2T_^H0[;k^
0:BniP2/_^h0Y;kV0:bnkP<0000000T0[<K^0:C6kP000000Y<;V0:c2kP2T`^H0[<;^0:C2iP2/`^h0
0`0000060:C6kP<0000000T0Y<;^0:C6kP2T`^h0Y<K^0:C2kP2Ta^h0Y<;^0:C6kP2T`^h01@000000
0`2Ta^h000000:C6kP040:C6kP<0000000L0Y<;^0:C6kP2T`^h0Y<K^0:C2kP2Ta^h0Y<;^00800000
3`3oool200000100oooo000G0?ooo`80000001h0RefD08]MW02;GI@0RefL08]MU02;GI`0RefD08]M
W02;GI@0RefL08]MU02;GI`0RefD08]MW0000000Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:Ai
Y02TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`20000000;0;BE_@2dVKd0]9Fm0;BI_@2dUKd0]9Vm
0;BE_@2dVKd0]9Fm0;BI_@2dUKd00P0000001`3=k^h0cN[^0000002m^]D0_K[N0;fje@2m^]h00P00
00000`2Ll^H0W?KV09cbiP03000000<0RgVm1@0000004`2DPKd0W86m09B1_@2LPKd0U86m09b1_@2D
PKd0W86m09B1_@2LPKd0U86m09b1_@2DPKd0000009BEe@2LVMD0U9GE09bIe@2DUMD00`0000003P2L
SKd0Y8g509b=_@2TSLD0W8fm0000002LVLd0Y9gE09bIc@2TWMD0W9W=0:BMe@2LVLd0Y9gE0`000000
1@2TVLd0Y9G50:BIc@2TULD0000000P0Y:7E0`0000002P2/YMD0Y:GE0:bUe@2TYMD000000:BZe@2/
Z]h0Y:[E0:bZgP2TZ]D30000000:0:b^gP2//]h0[:kN0:bbgP2/[]h0[;;N0:b^gP000000Y;;N0:bb
iP<000002P2/^^H3000000090:bniP2/`^h0[;kV0:c2kP2/_^H0[<;^0:bniP2/`^h0[;kV00800000
00P0[<[f0:C6kP2/b_H0Y<K^0:c:mP000000[<;^0:C2kP@0000000H0[<Kf0:C6kP2/a_H0Y<K^0:c6
mP2Ta^h3000000090:C6kP2Tb_H000000:c6mP2Ta^h0[<Kf0:C6kP2/a_H0Y<K^00<0000000T0Y<K^
0:C6mP2Ta^h0Y<Kf0000002Ta_H000000:c6mP2Ta^h00`0000001P2Ta^h0[<Kf0:C6kP2/a_H0Y<K^
0:c6mP@0000000<0Y<;^0:c6kP000000403oool00`000000oooo0?ooo`0=0?ooo`005`3oool20000
00d0RefD0P0000008`2LNJ@0Y7fT09aiY02TOJ@0W7VT0:AmY02LNJ@0Y7fT09aiY02TOJ@0W7VT0:Am
Y02LNJ@0Y7fT09aiY0000000[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE
_@000000cN[V0<g^kP3=j^H0cNk^0<gZiP000000_K[E0080000000<0W?KV09cbgP2Lm^H00P000000
0`23NK@0RgVm000000050000000309AeY02LMJ`0000000d0U86m00<0000009bIe@2DULd00`000000
3P2LRKd0W8g509b9_@2LSLD0W8Vm09b=a@2LRKd0W8g509b9_@000000W9W=09bMe@2LVLd0W9gE0`00
00003@2TVLd0Y9G50:BIc@2TULD0Y9W=0:BEa@2TVLd0Y9G50000002LXLd0Y:7E09bQc@2TXMD00`00
00002`2/YMD0Y:7=0:bUe@2TXLd0[:GE0:BQc@2/YMD0Y:7=0000002TZ]D0Y:[N0080000000X0Y:kE
0:bbgP2T[]D0[;;N0:B^e@2//]h0Y:kE0:bbgP2T[]D0[;;N0`0000002`2/^^H0Y;[N0:bjiP2T^]h0
[;[V0:BjgP2/^^H0Y;[N0:bjiP2T^]h0[;[V0080000000T0Y<K^0:c6kP000000[;k^0:BniP2/_^h0
Y;kV0:bnkP2T_^H00`0000001`2Ta^h0[<K^0:C6kP2/a^h0Y<K^0:c6kP2Ta^h00`0000080:C6kP80
000000P0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP000000Y<;^0:C6kP@00000202Ta^h4000000060:C2kP2T
a^h0Y<;^0:C6kP2T`^h0Y<K^0`0000070:C2kP030000003oool0oooo00d0oooo00<000000?ooo`3o
ool03@3oool001L0oooo0P0000007@2;GI@0RefL08]MU02;GI`0RefD08]MW02;GI@0RefL08]MU02;
GI`0RefD08]MW0000000[86/0000002TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0Y7VT0:Am[02T
NJ@0Y7f/0:AiY02TOJ`0Y7VT008000000180]9Vm0;BE_@2dVKd0]9Fm0;BI_@2dUKd0]9Vm0;BE_@2d
VKd0]9Fm0000003=j^h0cNk^0<gZkP3=k^h0cN[^0<g^kP3=j^h60000000308]i_@000000000000@0
000001<0U7F/09ae[02DMJ`0W7F/09Ae[02LMJ`0000009b1_@2DPKd0W86m09B1_@2LPKd0U86m09b1
_@2DPKd0W86m09B1_@2LPKd0U86m00<0000000l0W8fm0:B=a@2LSKd0Y8g509b=_@2TSLD0W8fm0:B=
a@2LSKd0Y8g509b=_@2TSLD000000:BMe@2LVLd00P000000302TVLd0Y9G50:BIc@2TULD0Y9W=0:BE
a@2TVLd0Y9G50:BIc@2TULD0Y9W=00000080Y:7E0P0000002`2/YMD0Y:GE0:bUe@2TYMD0[:GE0:BU
e@2/YMD0Y:GE0:bUe@2TYMD0[:GE00<0000000X0[;;N0:b^gP2//]h0[:kN0:bbgP2/[]h0[;;N0:b^
gP2//]h0[:kN0P0000000`2d`^h0[;kV000000080:bjiP<000001@2/a^h00`000000[<;^0:bniP03
000000060:C6kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f10000000202Tb_H0[<kf0000002/a_H0Y<K^0:c6
mP2Ta^h0[<Kf0`0000001`2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP2Tb_H00`0000002P2Ta_H0Y<K^
0:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP2Ta^h4000000040:C6mP000000[<Kf0:C6kP@00000
00X0[<K^0:C2kP2/a^h0Y<;^0:c6kP2T`^h0[<K^0:C2kP2/a^h0Y<;^0P00000?0?ooo`030000003o
ool0oooo00`0oooo000F0?ooo`030000003oool0000000/0RefD01`000000:b1Y02/PJ@0000009ai
Y02TOJ@0W7VT0:AmY02LNJ@0Y7fT09aiY02TOJ@0W7VT0:AmY02LNJ@0Y7fT09aiY0000000aJk50<Fb
a@000000]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0P0000001P3=j^H0cNk^0<gZiP3=k^h0cN[V
0<g^kP<0000000@0RfVT08]UY02;JJ@0RfFT1@0000002P2DMJ@0W7F/09AeY02LMJ`0U7FT09ae[02D
MJ@0W7F/09AeY000000:09B1_@80000000l0W86d0000002LRKd0W8g509b9_@2LSLD0W8Vm09b=a@2L
RKd0W8g509b9_@2LSLD0W8Vm09b=a@2LRKd00`0000003@2TULD0Y9W=0:BEa@2TVLd0Y9G50:BIc@2T
ULD0Y9W=0:BEa@2TVLd0Y9G50:BIc@2TULD00`0000002`2TXLd0[:GE0:BQc@2/YMD0Y:7=0:bUe@2T
XLd0[:GE0:BQc@2/YMD0Y:7=0080000000X0[;;N0000002//]h0Y:kE0:bbgP2T[]D0[;;N0:B^e@2/
/]h0Y:kE0`0000040:bniP060000002/^^H0Y;[N0:bjiP2T^]h0[;[V0`000000202/a^h0Y<K^0:c6
kP2Ta^h0[<K^0:C6kP2/a^h0Y<K^0`0000001P2Ta^h0[<K^0:C6kP2/a^h0Y<K^0:c6kP<0000000L0
Y<kf0:C:kP2Tc_H0Y<[^0:C>mP2Tb^h000000080Y<K^0`0000001`2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f
0:C6kP2Tb_H00`0000000`2Ta_H0Y<K^000000060:C6kP@0000000D0Y<;^0:C6kP2T`^h000000:C2
kP03000000X0Y<;^100000000`2T_^H0Y;k^0000000?0?ooo`030000003oool0oooo00/0oooo000F
0?ooo`@0000001T0RefL08]MU02;GI`0RefD08]MW02;GI@0RefL08]MU02;GI`000000:b1[02/PJ@0
[86/0:b1Y0000000Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@0Y7f/008000000140
aJk50<Fbc@35[/D000000;BE_@2dVKd0]9Fm0;BI_@2dUKd0]9Vm0000003=k^h0cN[^0<g^kP3=j^h0
cNk^0<gZkP02000000D0RfVT10000000502DMJ`0W7F/09Ae[02LMJ`0U7F/09ae[02DMJ`0W7F/09Ae
[02LMJ`0U7F/09ae[0000000W86m09B1_@2LPKd0U86m09b1_@2DPKd0W86m0`0000003`2TQK@0W86d
0:B5]02LPK@0000009b=_@2TSLD0W8fm0:B=a@2LSKd0Y8g509b=_@2TSLD0W8fm0:B=a@020000000>
0:BA_@000000Y9G50:BIc@2TULD0Y9W=0:BEa@2TVLd0Y9G50:BIc@2TULD0Y9W=0:BEa@2TVLd20000
000<0:bQc@2/YMD000000:bUe@2TYMD0[:GE0:BUe@2/YMD0Y:GE0:bUe@2TYMD0[:GE0P0000002@2/
/]h0];KN0:bbgP2d]]h000000:bbgP2/[]h0[;;N0:b^gP03000000090:bniP2d`^h0[;kV0;C2kP2/
_^H0]<;^0:bniP2d`^h000000080[;[V0P0000080:c6kP<0000000D0Y<kf0:cBmP000000[<[f0:C6
kP04000000090:c>mP2Tb_H0[<kf0:C:mP2/c_H0Y<[f0:c>mP2Tb_H0[<kf00<0000000H0Y<[f0:C6
kP2Tb_H0Y<K^0:C:mP2Ta^h4000000090:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0000002Ta^h0Y<Kf
00<0000000L0Y<Kf0:C2kP2Ta_H0Y<;^0:C6mP2T`^h0Y<Kf0080000000X0Y;k^0:c2kP000000[<K^
0:C2kP2/a^h0Y<;^0:c6kP2T`^h0[<K^0`0000001`2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP000000
3P3oool2000000T0oooo0`0000010?ooo`005P3oool010000000oooo0?ooo`00000808]MU0030000
002/PJ@0[86T00<0[86T01`000000:AmY02LNJ@0Y7fT09aiY02TOJ@0W7VT0:AmY02LNJ@0Y7fT0000
0035//D0aJk50<Fba@35[/D0aK;50<F^a@000000[9Fm0;BI_@2/UKd0]9Vm0000003=k^h0cN[V0<g^
kP3=j^H0cNk^0P0000001P2;IJ@0RfVT08]UY02;JJ@0RfFT08]YY08000000100U6fL0000002DMJ@0
W7F/09AeY02LMJ`0U7FT09ae[02DMJ@0W7F/09AeY02LMJ`0U7FT09ae[02DMJ@00000102DPKd30000
00L0W86d00T0000009b=a@2LRKd0W8g509b9_@2LSLD0W8Vm09b=a@2LRKd00P0000003P2TTLD0Y96m
0:BAa@2TTKd000000:BEa@2TVLd0Y9G50:BIc@2TULD0Y9W=0:BEa@2TVLd0Y9G50P000000302/YLd0
[:7=0:bUc@2/XLd000000:BQc@2/YMD0Y:7=0:bUe@2TXLd0[:GE0:BQc@80000000T0[;;N0:bbe@2/
/]h0[;;E0:bbgP2//]D000000:B^e@2//]h00P00000;0:bniP<0000000H0Y<K^0:c6kP2Ta^h0[<K^
0:C6kP2/a^h4000000050:C>mP2Td_H0Y<kf0:CBmP2Tc_H00`0000002P2Tb^h0Y<kf0:C:kP2Tc_H0
Y<[^0:C>mP2Tb^h0Y<kf0:C:kP2Tc_H3000000060:C:mP2Lb_H000000000002Tb_H0Y<K^0`000000
2P2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H3000000070:C6kP2T`^h0Y<K^
0:C2kP2Ta^h0Y<;^0:C6kP03000000060:C2kP2T_^H000000:BniP2T`^h000000P2T`^h400000009
0:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^008000003`3oool00`000000oooo0?oo
o`070?ooo`4000000@3oool10?ooo`005P3oool09P000000oooo0?ooo`000000RefL08]MU02;GI`0
RefD08]MW02;GI@0RefL0000002/PJ`0[86T0:b1[02/PJ@0[86/0:b1Y0000000Y7VT0:Am[02TNJ@0
Y7f/0:AiY02TOJ`0Y7VT0:Am[0000000aK;=0<F^a@35//d0aJk50<Fbc@35[/D0aK;=0000002dVKd0
]9Fm0P000000103=k^h0cN[^0<g^kP3=j^h2000000D0RfVT0`000000502LLJ@0W6fL09aaY02LKI`0
000009Ae[02LMJ`0U7F/09ae[02DMJ`0W7F/09Ae[02LMJ`0U7F/09ae[02DMJ`0W7F/09Ae[0000000
U86m0P0000004@2TQK@0W86d0:B5]02LPK@0Y8Fd09b1]02TQK@0W86d0:B5]02LPK@0000009b=_@2T
SLD0W8fm0:B=a@2LSKd0Y8g50080000000h0Y96m0:bAa@2TTKd0[9750:BA_@2/TLD000000:BIc@2T
ULD0Y9W=0:BEa@2TVLd0Y9G50:BIc@80000000`0[:7=0:bUe@2/XLd0[:GE0:bQc@2/YMD0[:7=0000
002TYMD0[:GE0:BUe@2/YMD2000000080:bbgP2d]]h0[;;N0;BfgP2//]h0];KN0:bbgP2d]]h30000
000:0;C2kP2/_^H0]<;^0:bniP2d`^h0[;kV0;C2kP2/_^H0]<;^0:bniP<000000P2/d_H200000080
[<K^0`0000001P2/d_H0Y<kf0:cBmP2Tc_H0[=;f0:C>mP<0000000T0W=;f0:CFo`000000[<kf0:C:
mP2/c_H0Y<[f0:c>mP2Tb_H0100000001P2Tc_H0Y<[f0:C>mP2Tb_H0Y<kf0:C:mP<0000000T0Y<K^
0:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP2Ta^h0100000020:C2kP050000002T`^h0Y<Kf0:C2
kP2Ta_H00`000000202/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP000000Y;k^0`0000002@2T_^h0[<;^
0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP04000000030:bnkP2T^^H0000000l0oooo00<00000
0?ooo`3oool00`3oool500000040oooo000F0?ooo`@000001P2;GI@00`000000[86T0:b1Y0050:b1
Y00B0000002TOJ@0W7VT0:AmY02LNJ@0Y7fT09aiY02TOJ@000000<Fba@35[/D0aK;50<F^a@35//D0
aJk50<Fba@35[/D0aK;50P000000103=j^H0cNk^0<gZiP3=k^h30000000308]YY02;IJ@0RfVT00@0
000001@0U6fL09aaY02DKI`0W76T09A]W02LLJ@0000009ae[02DMJ@0W7F/09AeY02LMJ`0U7FT09ae
[02DMJ@0W7F/09AeY02LMJ`0U7FT09ae[0800000302LPK@01@000000W8g509b9_@2LSLD0W8Vm0080
000000h0Y9750:BA_@2TTLD0Y96m0:BAa@2TTKd0Y9750:BA_@2TTLD000000:BIc@2TULD0Y9W=0:BE
a@80000000`0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd000000:bUe@2TXLd20000
00080:bbgP2//]D0[;;N0:bbe@2//]h0[;;E0:bbgP2//]D2000000030;C>kP2/c^h0000000H0[;kV
100000001P2Td_H0[=;f0:CBmP2/d_H0Y=;f0:cBmP<0000000D0Y=;f0:C>mP2Td_H0Y<kf0:CBmP04
0000000909cBmP2Td_l0W=;f0:CBo`2Ld_H0Y=;o0000002Tc_H0Y<[^00<0000000H0W<[f0:C:mP2L
b_H0Y<[f09c:mP2Tb_H40000000909c2kP2Ta_H000000:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^00<0
000000T0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h000000:C6kP2T`^h00P0000001`2T`^h0Y;kV0:C2
kP2T_^H0Y<;^0:BniP2T`^h0100000002@2T^^H0Y;k^0000002T_^h0Y;kV0:BnkP2T_^H0Y;k^0:Bn
iP03000000070:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0000000?0?ooo`030000003oool0oooo00<0
oooo00@000000?ooo`3oool000000P3oool001H0oooo02@000000?ooo`3oool0000008]MW02;GI@0
RefL08]MU02;GI`0RefD0000002/PJ@0[86/0:b1Y02/PJ`0[86T0:b1[02/PJ@000000:AiY02TOJ`0
Y7VT0:Am[02TNJ@0Y7f/00000035//d0aJk50<Fbc@35[/D0aK;=0<F^a@35//d0aJk50<Fbc@35[/D2
000000070<g^kP3=j^h000000000002DII`0Rf6L09AUW0030000000F08]YY0000000000009a]W02L
LJ@0W6fL09aaY02LKI`0W76T09a]W0000000U7F/09ae[02DMJ`0W7F/09Ae[02LMJ`0U7F/09ae[02D
MJ`0W7F/09Ae[0<000000100W86d0:B5]02LPK@0Y8Fd09b1]02TQK@0W86d0:B5]02LPK@0Y8Fd09b1
]02TQK@0W86d0:B5]0000000Y8g50P0000003P2TTKd0[9750:BA_@2/TLD0Y96m0:bAa@2TTKd0[975
0:BA_@2/TLD0Y96m0000002TULD0Y9W=0P0000002`2/XLd0[:GE0:bQc@2/YMD0[:7=0:bUe@2/XLd0
[:GE0:bQc@2/YMD0[:7=00<0000000P0[;;N0;BfgP2//]h0];KN0:bbgP2d]]h0[;;N0;BfgP800000
00P0[<k^0;C>mP2/c^h0]<kf0:c>kP000000[;kV0;C2kP<000001`2/d_H3000000050:CNo`2Lf_l0
00000:C>mP2/d_H00`0000002@2Te_l0W=;f0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0W=;f0:CFo`040000
00060:C:mP2Tc_H0Y<[f0:C>mP2Tb_H0Y<kf0`0000002@2Ta_H0Y<;^0:C6mP2T`^h0Y<Kf0:C2kP00
0000Y<K^0:C6mP04000000T0Y<;^0`0000001P2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP<0000000T0
[;k^0:BjiP2/_^h000000:bnkP2T^^H000000:BnkP2/`^h0100000002@2T^^H0[;k^0:BjiP2/_^h0
Y;[V0:bnkP2T^^H0[;k^0:BjiP02000000h0oooo0P0000050?ooo`030000003oool000000080oooo
00030?ooo`800000103oool2000000@0oooo0P0000050?ooo`040000003oool0oooo000000D0RefD
00<000000:b1Y02/PJ@01P2/PJ@040000000Y7fT09aiY02TOJ@0W7VT0:AmY0000000aK;50<F^a@35
//D0aJk50<Fba@35[/D0aK;50<F^a@35//D60000000308]QU02DII`0000000<0000001`0RfFT08]Y
Y0000000W76T09A]W02LLJ@0U6fL09aaY02DKI`0W76T09A]W02LLJ@0000009ae[02DMJ@0W7F/09Ae
Y02LMJ`0U7FT09ae[02DMJ@0W7F/09AeY0000000W7f/0:B1[02LOJ`000003@2LPK@20000000>0:BA
a@2TTKd0Y9750:BA_@2TTLD0Y96m0:BAa@2TTKd0Y9750:BA_@2TTLD0Y96m0:BAa@2TTKd20000000;
0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd00P0000002@2d_]h00000
0:bbe@2//]h0[;;E0:bbgP2//]D0[;;N0:bbe@02000000070;C>kP2/c^h0]<k^0:c>kP2dc^h0[<k^
0;C>kP03000000060:cBmP2Td_H0[=;f0:CBmP2/d_H0Y=;f100000001@2Lg_l0W=[f09cNo`2Lf_H0
W=ko00<0000000P0Y=;o09cBmP2Td_l0W=;f0:CBo`2Ld_H0Y=;o09cBmP@0000000H0W<kf09c:mP2L
c_H000000:C:mP2Lb_H4000000090:C6mP2L`^h0Y<Kf09c2kP2Ta_H0W<;^0:C6mP2L`^h0Y<Kf00<0
000000T0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0100000001@2T^^H0Y;k^0000
002T`^h0Y;kV00@0000000T0Y;k^0:BjiP2T_^h0Y;[V0:BnkP2T^^H0Y;k^0000002T_^h00`000000
2@2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP04000000030:BjiP2/^^H0000000X0
oooo1@0000060?ooo`8000000P3oool00080oooo00@000000?ooo`3oool00000203oool010000000
oooo0?ooo`0000030?ooo`<0000001@0oooo0000002;GI`0RefD08]MW02;GI@000000:b1Y02/PJ`0
[86T0:b1[02/PJ@0[86/0:b1Y02/PJ`0[86T0:b1[0000000Y7f/0:AiY080000000T0aK;=0<F^a@35
//d0aJk50<Fbc@35[/D0aK;=0<F^a@35//d01@0000001`2DII`00000000000000000U6FL08]QW02D
II`00`0000004`2LLJ@0W6fL09aaY02LKI`0W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T0000002L
MJ`0U7F/09ae[02DMJ`0W7F/09Ae[02LMJ`00P000000402TOJ`0Y86/0:Am[02TPJ`000000:B5]02L
PK@0Y8Fd09b1]02TQK@0W86d0:B5]02LPK@0Y8Fd09b1]02TQK@30000000=0:bAa@2TTKd0[9750:BA
_@2/TLD0Y96m0:bAa@2TTKd0[9750:BA_@2/TLD0Y96m0:bAa@020000000;0;BZc@000000[:7=0:bU
e@2/XLd0[:GE0:bQc@2/YMD0[:7=0:bUe@2/XLd00P0000002@2d`^H0];kV0;C2iP2d_^H000000:bb
gP2d]]h0[;;N0;BfgP02000000070:c>kP2dc_H0[<k^0;C>mP2/c^h0]<kf0:c>kP02000000040:CV
o`2Lh_l0000000000080[=;f0`0000001P2Lf_l0Y=ko09cJo`2Tg_l0W=[o0:CNo`<0000000L0W=[o
09CFo`00000000000:CFo`2Ld_H0Y=Ko00@0000000H0W<[f0:C>o`2Lb_H0Y<ko09c:mP2Tc_l40000
000:0:C2kP2Ta_H0Y<;^0:C6mP2T`^h0Y<Kf0:C2kP2Ta_H0Y<;^0:C6mP<0000000<0Y;k^0:BjiP00
00001P2T`^h3000000060:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V0`0000002@2T^^H0[;k^0:BjiP2/
_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP04000000090:bnkP2T^^H000000:BjiP2/_^h0Y;[V0:bnkP2T
^^H0[;k^00<0000000L0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0000000H0oooo100000040?ooo`03
0000003oool0oooo00P0oooo00020?ooo`040000003oool0oooo000000/0oooo00D000000?ooo`3o
ool0oooo000000030?ooo`050000002;GI@0RefD08]MU00000002`2/PJ@03@000000W7VT00000035
[/D0aK;50<F^a@35//D0aJk50<Fba@35[/D0aK;50<F^a@35//D00`0000009`2DII`0000000000000
0000U6FL08]QU02DII`0Rf6D09AUW02;HI@0000009A]W02LLJ@0U6fL09aaY02DKI`0W76T09A]W02L
LJ@0U6fL09aaY02DKI`0W76T09A]W0000000U7FT09ae[02DMJ@0W7F/09AeY02LMJ`000000:B1[02L
OJ`0Y86/09am[02TPJ`0W7f/0000000:09b1]00?0000002/TKd0[96m0:bA_@000000Y9750:BA_@2T
TLD0Y96m0:BAa@2TTKd0Y9750:BA_@2TTLD0Y96m0080000000/0]:[=0:bZc@2dZ/d000000:bUc@2/
XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@02000000090;BngP2d`^H0];kN0;C2iP2d_]h0]<;V0000002/
/]h0[;;E0080000000H0]<k^0:c>kP2dc^h0[<k^0;C>kP2/c^h30000000509cRmP2Lh_l0W>;f09cR
o`2Lh_H00`0000001@2Lf_H0W=ko09cJmP2Lg_l0W=[f00@0000000H0W=Ko09CFo`2Le_l0U=Ko09cF
o`2De_l40000000609c>mP2Lb_H0W<kf09c:mP2Lc_H0W<[f100000002P2T_^h0W;k^0:BnkP000000
Y<Kf09c2kP2Ta_H0W<;^0:C6mP2L`^h4000000080:BnkP2T^^H0Y;k^0:BjiP2T_^h000000:C2kP2T
_^h4000000060:BnkP2T^^H0Y;k^0:BjiP2T_^h0Y;[V0`0000002@2T]^H0Y;[V0000002T_^h0Y;[V
0:BnkP2T^^H0Y;k^0:BjiP03000000090:BjiP2/^^H0Y;[V0000002T^^H0[;[V0000002/_^H0Y;[V
00@0000000T0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H00P0000020?ooo`<00000
203oool00`000000oooo0?ooo`080?ooo`000P3oool010000000oooo0?ooo`0000090?ooo`800000
103oool050000000oooo0?ooo`3oool0000008]MW02;GI@000000:b1Y02/PJ`0[86T0:b1[02/PJ@0
[86/0:b1Y02/PJ`0[86T0:b1[02/PJ@0[86/0P0000002@35[/D0aK;=0<F^a@35//d0aJk50<Fbc@35
[/D0aK;=0<F^a@020000000<09AUW000000000000000002DII`0Rf6L09AUW02;HI`0U6FL08]QW02D
II`0Rf6L0P0000004P2LLJ@0W6fL09aaY02LKI`0W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T09a]
W02LLJ@0000009ae[02DMJ`0W7F/09Ae[08000000100Y86/0:Am[02TPJ`0Y7f/0:B1[02TOJ`0Y86/
0000002TQK@0W86d0:B5]02LPK@0Y8Fd09b1]02TQK@0W86d0P0000003@2/UKd0[96m0:bE_@2/TKd0
00000:BA_@2/TLD0Y96m0:bAa@2TTKd0[9750:BA_@2/TLD00P0000002`2dZ/d0]:[E0;BZc@2dZ]D0
]:[=0;BZe@000000[:GE0:bQc@2/YMD0[:7=0080000000P0]<;V0;BniP2d`^H0];kV0;C2iP2d_^H0
]<;V0;BniP<0000000D0[<k^0;C>mP2/c^h0]<kf0:c>kP030000000509cRo`2Ti_l0W>;o0:CVo`2L
h_l00`0000001@23h_l0Rn;o0000002Tg_l0W=[o00<0000000L0U=Ko09cJo`2De_l0W=[o09CFo`2L
f_l0U=Ko00<0000000H0W<Kf09c:mP000000Y<ko09c:mP2Tc_l40000000;0:BnkP2T`^h0Y;k^0:C2
kP2T_^h0Y<;^0:BnkP2T`^h000000:C6mP2T`^h00`000000202T^^H0Y;k^0:BjiP2T_^h0Y;[V0:Bn
kP2T^^H0Y;k^10000000202T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0P000000202T^^H0
[;[V0:BjiP2/^^H0Y;[V0000002T^^H0[;k^100000002@2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T
^^H000000:BjiP03000000090:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V00@00000
00@0[;k^0:BniP0000000000303oool00`000000oooo0?ooo`070?ooo`000P3oool010000000oooo
0?ooo`0000090?ooo`030000003oool0oooo00<0oooo0P0000020?ooo`030000002;GI@0000000d0
[86T00`000000<F^a@35//D0aJk50<Fba@35[/D0aK;50<F^a@35//D0aJk50000003oool20000000Z
09AUW02;HI@0U6FL08]QU02DII`0Rf6D09AUW02;HI@0U6FL08]QU0000000W6fL0000002DKI`0W76T
09A]W02LLJ@0U6fL09aaY02DKI`0W76T09A]W02LLJ@0U6fL09aaY02DKI`0000009AeY02LMJ`0U7FT
0000002LOJ`0Y86/09am[02TPJ`0W7f/0:B1[02LOJ`0Y86/09am[02TPJ`000001P2LPK@00`000000
[96m0:bA_@050:bA_@060000002TTKd0Y9750:BA_@2TTLD0Y96m0P0000002`2dZ/d0[:[=0;BZc@2/
Z/d0]:[=0:bZc@2dZ/d0[:[=0000002/XLd0[:G=0080000000P0];kN0;C2iP2d_]h0]<;V0;BngP2d
`^H0];kN0;C2iP80000000H0Y>Kf0:CVo`00000000000:c>kP2dc^h20000000409cRmP2Lh_l0W>;f
09cRo`@0000000@0Pmkf08?Ro`23g_H0Pn;o100000001P2De_l0W=Ko09CFo`2Le_l0U=Ko09cFo`@0
000000D0W<Kf09c:mP2La_H0W<[f09c6mP040000000;0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:Bn
kP2L_^h0Y;k^09bnkP2T_^h0100000001`2T^^H0Y;k^0:BjiP2T_^h0Y;[V0:BnkP2T^^H010000003
0:BfiP060000002T_^h0Y;[V0:BnkP2T^^H0Y;k^0`0000001`2T^^H0Y;KV0:BjiP2T]^H0Y;[V0:Bf
iP2T^^H00`0000002P2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H300000009
0:BjiP2/_^H000000:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V00<0000000L0Y;[V0:bnkP2T^^H0[;k^
0:BjiP2/_^h0000000d0oooo00<000000?ooo`3oool01`3oool000<0oooo0P00000:0?ooo`<00000
0`3oool010000000oooo0?ooo`3oool20000000F0:b1Y02/PJ`0[86T0:b1[02/PJ@0[86/0:b1Y02/
PJ`0[86T0:b1[02/PJ@0[86/0:b1Y0000000g_o=00000035[/D0aK;=0<F^a@35//d0aJk50<Fbc@@0
000001d0oooo0000002;HI`0U6FL08]QW02DII`0Rf6L09AUW02;HI`0U6FL08]QW0000000W76L0:Aa
W0000000W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T09a]
W0030000000@0:Am[02TPJ`0Y7f/0:B1[02TOJ`0Y86/0:Am[02TPJ`0Y7f/0:B1[02TOJ`0000009b1
]02TQK@0W86d0:B5]080000000`0[96m0:bE_@2/TKd0[9Fm0:bA_@2/UKd0[96m0:bE_@000000[975
0:BA_@2/TLD20000000:0;BZc@2dZ]D0]:[=0;BZe@2dZ/d0]:[E0;BZc@2dZ]D0]:[=0;BZe@<00000
00L0]<;V0;BniP2d`^H0];kV0;C2iP2d_^H0]<;V00<0000000D0[>[o0:CVo`2/j_l0Y>Ko0:cZo`03
000000030:CVo`2Lh_l0Y>Ko00<0000000@0Rn;o08?Ro`2;h_l0Pn;o100000000`2De_l0RmKo09CF
o`020000000309CFo`2Lf_l000000080000000H0W<[f09c6mP2Lb_H0W<Kf09c:mP2La_H30000000;
09bfiP2T]^h000000:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0100000001`2T]^H0Y;;V
0:BfiP000000Y;k^0:BjiP2T_^h0100000002P2T]^H0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H00000
0:BjiP2/_^h3000000080:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP2T^^H20000000;0:bjiP00
0000[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h00P000000202T^^H0[;k^0000002/
_^h0Y;[V0000002T^^H0[;k^100000002@2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:Bn
iP02000000`0oooo00<000000?ooo`3oool01`3oool001D0oooo00D000000?ooo`3oool0oooo0000
000=0:b1Y0080000003Eold0g_o=00000035//D0aJk50<Fba@35[/D3000000@0oooo02/0000009AU
W02;HI@0U6FL08]QU02DII`0Rf6D09AUW02;HI@0000009a]W02TLI`0W6fL0000002DKI`0W76T09A]
W02LLJ@0U6fL09aaY02DKI`0W76T09A]W02LLJ@0U6fL09aaY02DKI`0W76T0000002TPJ`0W7f/0:B1
[02LOJ`0Y86/09am[02TPJ`0W7f/0:B1[02LOJ`0Y86/09am[02TPJ`0000000<0W86d00<000000:bA
_@2/TKd02@2/TKd30000000:0;BZc@2/Z/d0]:[=0:bZc@2dZ/d0[:[=0;BZc@2/Z/d0]:[=0:bZc@80
000000L0_N;f0;CNkP000000];kN0;C2iP2d_]h0]<;V00<0000000D0Y>Ko0:CVmP2Ti_l0Y>Kf0:CV
o`0300000080H][N100000000`23h_l0Pmkf08?Ro`04000000H0RmKo1@0000001@2Lb_H0W<Kf09c:
mP2La_H0W<[f00@0000000T0W;KV0:BfiP2L]^H0Y;KV09bfiP000000W;k^0:BnkP2L_^h01@000000
1`2T/]h0Y;KV0:BbgP2T]^H0Y;;N0:BfiP2T/]h01000000;0:BfiP<0000000P0Y;[V0:BfiP2T^^H0
Y;KV0:BjiP2T]^H0Y;[V0:BfiP<0000000X0Y;[V0:bjiP2T^^H0[;[V0000002/^^H0Y;[V0:bjiP2T
^^H0[;[V0`0000001`2/_^H0Y;[V0:bniP2T^^H0[;kV0000002/_^H00`0000002@2/_^h0Y;[V0:bn
kP2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP04000000030:BniP2T`^h0000000/0oooo00<000000?oo
o`3oool01`3oool001@0oooo00<000000?ooo`0000000P3oool03P000000[86/0:b1Y02/PJ`0[86T
0:b1[02/PJ@0[86/0:b1Y02/PJ`0[86T0:b1[02/PJ@00000103Nold5000000H0oooo01/0000008]Q
W02DII`0Rf6L09AUW02;HI`0U6FL08]QW0000000W76L0:AaW02LLI`0Y76L09aaW0000000W6fL09aa
Y02LKI`0W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T09a]W02LLJ@00P0000003P2TOJ`0Y86/0:Am
[02TPJ`0Y7f/0:B1[02TOJ`0Y86/0:Am[02TPJ`0Y7f/0:B1[02TOJ`0Y86/0`0000005`2/UKd0[96m
0:bE_@2/TKd0[9Fm0:bA_@2/UKd0[96m0:bE_@2/TKd0[9Fm0:bA_@000000]:[=0;BZe@2dZ/d0]:[E
0;BZc@2dZ]D0]:[=0;BZe@2dZ/d0]:[E0080000000L0]=kf0;gRmP2dg_H0_N;f0000002d`^H0];kV
0080000000@0[>[o0:CVo`2/j_l0Y>Ko100000000`1Zf]h00000000000020000000305;>iP1Jd^H0
000000@0000000@0U=Ko08_Fo`2De_l0RmKo1P0000001`2L__H0W;k^09bnmP2L_^h0000009c6mP2L
b_H00`0000002@2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP040000000:0:BbiP2T
]^H0Y;;V0:BfiP2T/^H0Y;KV0:BbiP2T]^H0Y;;V0:BfiP80000000`0Y:kN0000002T]^H0[;[V0:Bf
iP2/^^H0Y;KV0:bjiP2T]^H0[;[V0:BfiP2/^^H3000000080:bfiP2T]]h000000:BjiP2/^^H0Y;[V
0:bjiP2T^^H30000000:0:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP000000Y;[V0:bnkP<00000
00L0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^00<0000000T0[<;^0:BniP000000Y;kV0:bnkP2T
_^H0[;k^0:BniP2/_^h00`0000001`2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2kP0000002`3oool00`00
0000oooo0?ooo`070?ooo`00503oool2000000<0oooo00<000000:b1Y02/PJ@02@2/PJ@010000000
eOo=0=koc@3Eold3000000030<F^a@000000000000L0oooo02`0000009AUW02;HI@0U6FL08]QU02D
II`0Rf6D0000002LKI`0Y76L09a]W02TLI`0W6fL0:AaW0000000W76T09A]W02LLJ@0U6fL09aaY02D
KI`0W76T09A]W02LLJ@0U6fL09aaY0000000[86/0000002TPJ`0W7f/0:B1[02LOJ`0Y86/09am[02T
PJ`0W7f/0:B1[02LOJ`0Y86/09am[02TPJ`0W7f/000000`0[96m0P0000002@2m`]D000000:bZc@2d
Z/d0[:[=0;BZc@2/Z/d0]:[=0:bZc@02000000060;gRmP2dg^h0_N;f0;CNkP2mh_H0]=k^0`000000
0`2Ti_H0Y>Ko0:CVmP030000000306;JgP000000000000P0000000<0Rlko08?:o`2;c_l01P000000
2@2D^^h0W;k^09BjkP2L_^h0U;[^09bnkP2D^^h0W;k^09BjkP04000000090:BfiP2L]^H0Y;KV09bf
iP2T]^H0W;KV0:BfiP2L]^H0Y;KV00<0000000/0Y:[N0:BZe@000000Y;;N0:BfiP2T/]h0Y;KV0:Bb
gP2T]^H0Y;;N0:BfiP03000000050:BbgP2T[]h0Y;;N0:B^gP0000001@2T]^H4000000090:bfiP2T
]]h0[;KV0:BfgP2/]^H0Y;KN0000002T]^H0Y;[V0080000000X0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/
^^H0Y;[V0:bjiP2T^^H0[;[V0`000000202/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0P00
0000202/_^h0Y;kV0:bnkP000000[;k^0000002/_^h0Y;[V100000002@2T`^h0Y;kV0:C2kP2T_^H0
Y<;^0:BniP2T`^h0Y;kV0:C2kP02000000X0oooo00<000000?ooo`3oool01`3oool001@0oooo00<0
00000?ooo`3oool00P3oool030000000[86/0:b1Y02/PJ`0[86T0:b1[02/PJ@0[86/0:b1Y02/PJ`0
[86T0:b1[0@000000`3Nold00`000000oooo0?ooo`070?ooo`0Z0000002;HI`0U6FL08]QW02DII`0
Rf6L0000002LLI`0Y76L09aaW02TLI`0W76L0:AaW02LLI`0000009a]W02LLJ@0W6fL09aaY02LKI`0
W76T09a]W02LLJ@0W6fL09aaY0000000[8F/0:b1Y0000000Y7f/0:B1[02TOJ`0Y86/0:Am[02TPJ`0
Y7f/0:B1[02TOJ`0Y86/0:Am[02TPJ`0Y7f/0P0000002P2/UKd0[96m0:bE_@2/TKd0[9Fm0:bA_@2/
UKd0[96m0:bE_@2/TKd2000000090<G6gP2m`]h0aLKN0000002dZ]D0]:[=0;BZe@2dZ/d0]:[E0080
000000H0]=kf0;gRmP2dg_H0_N;f0;CNmP2mh_H2000000<0B/Jd2`0000001@23c_l0Rlko08?>o`2;
c_l0Plko00@0000000/0W:[V09bZgP2LZ^H0000009bnmP2L_^h0W;kf09bnkP2L__H0W;k^09bnmP04
0000000:09bUgP2TZ]h0W:GN0000002L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP<0000000/0Y:kN0:BZ
gP2T[]h0Y:[N0:B^gP000000Y;KV0:BbiP2T]^H0Y;;V0:BfiP030000000:0:bbgP2T[]h0[;;N0:B^
gP2//]h0Y:kN0:bbgP000000[;[V0:BfiP<0000000X0Y;KN0:bfiP2T]]h0[;KV0:BfgP2/]^H0Y;KN
0:bfiP2T]]h0[;KV0`0000002@2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0:bjiP030000
00080:bnkP2T_^H000000:BjiP2/_^h0Y;[V0:bnkP2T^^H3000000070:BniP2/`^h0Y;kV0:c2kP2T
_^H000000:BniP03000000090:C2kP2/`^h0Y<;^0:c2kP2T`^h0[<;^0:C2kP2/`^h0Y<;^00@00000
00<0[<Kf0000003oool02@3oool00`000000oooo0?ooo`070?ooo`00503oool2000000@0oooo00<0
00000:b1Y02/PJ@01P2/PJ@3000000040=Goc@3Nold0eOo=0=koc@8000002P3oool0:0000000U6FL
08]QU02DII`0Rf6D0000002LKI`0Y76L09a]W02TLI`0W6fL0:AaW02LKI`0Y76L0000002LLJ@0U6fL
09aaY02DKI`0W76T09A]W02LLJ@0U6fL09aaY0000000[86/0:B1Y02/PJ`0Y86T0000002LOJ`0Y86/
09am[02TPJ`0W7f/0:B1[02LOJ`0Y86/09am[02TPJ`200000080]9fm00<000000:bA_@2/TKd01P2/
TKd02P000000_L;E0<G2gP2m`]D0aL;N0;g2e@35`]h000000;BZc@2/Z/d2000000060;gRmP2dg^h0
_N;f0;CNkP2mh_H0]=k^0P0000000`1:a[@00000000000040000000:08^bkP2;]_H0Rk;^08^fmP2;
/^h0RkKf0000002;c_l0Pl[o08_>o`@0000000/0U:GN09bZgP2DYMh0W:[N09BUgP2LZ]h0U:GN0000
002D^^h0W;k^09BjkP040000000;0:BZgP2LYMD0Y:[N09bUe@2TZ]h0W:GE0:BZgP2LYMD0000009bf
iP2T]^H00`000000302TZ]h0Y:[E0:BZgP2TZ]D0Y:[N0:BZe@2TZ]h0Y:[E0:BZgP000000Y;KV0:Bb
gP80000000X0Y;;N0:B^gP2T/]h0Y:kN0:BbgP2T[]h0Y;;N0:B^gP2T/]h0Y:kN0`0000002P2T]]h0
[;KV0:BfgP2/]^H0Y;KN0:bfiP2T]]h0[;KV0:BfgP2/]^H3000000080:bjiP2T^^H000000:BjiP2/
^^H0Y;[V0:bjiP2T^^H4000000080:bniP2T^^H0[;kV0:BjiP2/_^H000000:bniP2T^^H300000006
0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^100000020:C2kP070000002T`^h0Y;kV0:C2kP2T_^H0Y<;^
0:BniP03000000070:C2kP2Ta^h0Y<;^0:C6kP2T`^h0Y<K^000000090?ooo`030000003oool0oooo
00L0oooo000D0?ooo`030000003oool0oooo00<0oooo00H000000:b1Y02/PJ`0[86T0:b1[02/PJ@4
000000D0g_o=00<000000?ooo`3oool02P3oool0<`000000Rf6L09AUW02;HI`0000009aaW02TLI`0
W76L0:AaW02LLI`0Y76L09aaW02TLI`0W76L0000002LKI`0W76T09a]W02LLJ@0W6fL09aaY02LKI`0
W76T0000002/QJ`0[86T0:b5[02/PJ@0[8F/0000002TPJ`0Y7f/0:B1[02TOJ`0Y86/0:Am[02TPJ`0
Y7f/0:B1[0000000_Ig50;BM_@2mWLD0]9fm0000002/TKd0[9Fm0:bA_@2/UKd0[96m0:bE_@020000
00080;g2gP35a]h0_L;N0<G6gP2m`]h0aLKN0;g2gP35a]h2000000050;CNmP2mh_H0]=kf0;gRmP2d
g_H01`0000002P2;/^h0RkKf08^bkP2;]_H0Rk;^08^fmP2;/^h0RkKf08^bkP2;]_H50000000;09bZ
iP2LZ]h0W:[V09bZgP2LZ^H0W:[N09bZiP2LZ]h0W:[V09bZgP2LZ^H010000000302LYMh0Y:[N09bU
gP2TZ]h0W:GN0:BZgP2LYMh0Y:[N09bUgP2TZ]h0W:GN0:BZgP<0000000`0Y:kN0:BZgP2T[]h0Y:[N
0:B^gP2TZ]h0Y:kN0:BZgP2T[]h0Y:[N0:B^gP2TZ]h3000000090:B^gP2//]h0Y:kN0:bbgP2T[]h0
[;;N0:B^gP2//]h0Y:kN00<0000000T0Y;;N0:bfiP000000[;KV0:BfgP2/]^H0Y;KN0:bfiP2T]]h0
100000050:bjiP030000002/^^H0Y;[V00<0000000T0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV
0:bnkP2T_^H00`0000001P2T_^H0[<;^0:BniP2/`^h0Y;kV0:c2kP<0000000T0[<K^0:C2kP2/a^h0
Y<;^0000002T`^h000000:C2kP2/`^h010000000202T`^h0[<Kf0:C2kP2/a_H0Y<;^0:c6mP2T`^h0
[<Kf0P00000:0?ooo`030000003oool0oooo00H0oooo000D0?ooo`030000003oool0oooo00<0oooo
00@000000:b1Y02/PJ@0[86T0P0000020:b1Y0050000003Nold0eOo=0=koc@3Eold00P00000=0?oo
o`0E0000002DII`0Rf6D0000002LKI`0Y76L09a]W02TLI`0W6fL0:AaW02LKI`0Y76L09a]W02TLI`0
000009aaY02DKI`0W76T09A]W02LLJ@0U6fL008000000100[86/0:B1Y02/PJ`0Y86T0:b1[02TPJ@0
000009am[02TPJ`0W7f/0:B1[02LOJ`0Y86/09am[02TPJ`000001@2dWKd00`000000[96m0:bA_@03
0:bA_@0900000035`]h0_L;E0<G2gP2m`]D0aL;N0;g2e@35`]h0_L;E0080000000D0H]:d00000000
0000]=k^0;gRmP040000000908^=c@2DTLd0Rhg=09BAc@000000RkKf08^bkP2;]_H0Rk;^00H00000
00`0W97=09BAa@2LTLd0000009bZgP2DYMh0W:[N09BUgP2LZ]h0U:GN09bZgP2DYMh40000000=0:BI
c@2LULd0000009bUe@2TZ]h0W:GE0:BZgP2LYMD0Y:[N09bUe@2TZ]h0W:GE0:BZgP030000000<0:BQ
e@000000Y:[N0:BZe@2TZ]h0Y:[E0:BZgP2TZ]D0Y:[N0:BZe@2TZ]h0Y:[E0`000000202TZ]D0[:[E
0000002T/]h0Y:kN0:BbgP2T[]h0Y;;N100000002@2T/]h0[;;N0:BbgP2//]h0Y;;N0:bbgP000000
[;KV0:BfgP03000000090:BjiP2/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V00<0000000T0
Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H00`0000001@2T`^h0[<;^0000002/_^h0
Y;kV00@00000202T`^h4000000090:C6kP2T`^h0Y<K^0:C2kP2Ta^h0Y<;^0:C6kP2T`^h0Y<K^00<0
00000P2Ta^h00`000000oooo0?ooo`070?ooo`030000003oool0oooo00H0oooo000C0?ooo`030000
003oool0000000@0oooo10000000102/PJ`0[86T0:b1[00000040=koc@030000003oool0oooo00d0
oooo02@0000008]QW0000000W76L0:AaW02LLI`0Y76L09aaW02TLI`0W76L0:AaW02LLI`0Y76L09aa
W0000000W6fL09aaY02LKI`0W76T09a]W0000000[86T0:b5[02/PJ@0[8F/0:b1Y02/QJ`0[86T0:b5
[0000000Y86/0:Am[02TPJ`0Y7f/0:B1[02TOJ`20000000:0;fMa@2dWKd0_Ig50;BM_@2mWLD0]9fm
0000002/TKd0[9Fm0:bA_@80000000L0aLKN0;g2gP35a]h0_L;N0<G6gP2m`]h0aLKN0080000000<0
H]:/06;B]01RdZ`010000000202DTMD0Ri7=09BAe@2;TLd0U97E08^Ac@2DTMD0Ri7=1@0000003P2L
TLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0000009bZiP2LZ]h0W:[V10000000
3@2LVLd0Y9W=09bIc@2TVLd0W9W=0:BIc@000000Y:[N09bUgP2TZ]h0W:GN0:BZgP2LYMh010000000
2`2TXMD0Y:7=0:BQe@2TXLd0Y:7E0000002T[]h0Y:[N0:B^gP2TZ]h0Y:kN00@0000000P0Y:[E0:bZ
gP2TZ]D0[:[N0:BZe@000000Y:kN0:bbgP<0000000X0[;KV0:BbgP2/]^H0Y;;N0:bfiP2T/]h0[;KV
0:BbgP2/]^H0Y;;N0`0000090:bjiP<000000P2/`^h01P000000[;k^0:BniP2/_^h0Y;kV0:bnkP@0
000000D0Y<;^0:c6kP2T`^h0[<K^0:C2kP030000000:0:C2kP2/a^h0Y<;^0:c6kP2T`^h0[<K^0:C2
kP2/a^h0Y<;^0:c6kP<0000000P0[<Kf0:C6kP000000Y<;^0:c6mP2T`^h0[<Kf0:C2kP@0000000D0
[<Kf0:C6kP2/a_H0Y<K^0:c6mP02000000P0oooo00<000000?ooo`3oool01P3oool001<0oooo0P00
00030?ooo`<000001@2/PJ@01P000000g_o=0=Goc@3Nold0eOo=00000100oooo0P0000008P2LKI`0
Y76L09a]W02TLI`0W6fL0:AaW02LKI`0Y76L09a]W02TLI`0W6fL0:AaW0000000W76T09A]W02LLJ@0
U6fL0000002TPJ@0[86/0:B1Y02/PJ`0Y86T0:b1[02TPJ@0[86/0:B1Y0000000W7f/0:B1[02LOJ`0
Y86/09am[00000090;BM_@<0000000L0_L;E0<G2gP2m`]D0aL;N0;g2e@35`]h0_L;E0080000000<0
H]:d0000000000000`0000001`2DTLd0Rhg=09BAc@2;SLd0U97=08^=c@2DTLd0100000003`2DPK@0
000009BAa@2LTLd0U97509bAc@2DTLD0W97=09BAa@2LTLd0U97509bAc@2DTLD0W97=09BAa@040000
000>0:BIc@2LULd0Y9W=09bEc@2TVLd0W9G=0:BIc@2LULd0Y9W=09bEc@2TVLd000000:BZgP2LYMD3
0000000<0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0000002TZ]D0Y:[N0`000000
2@2/Z]D0Y:[E0:bZe@2TZ]D0[:[E0:BZe@2/Z]D0Y:[E0:bZe@030000000;0:bbgP2T/]h0[;;N0:Bb
gP2//]h0Y;;N0:bbgP2T/]h0[;;N0:BbgP2//]h00P000000202/^^H0[;kV0000002/^^H0Y;[V0:bj
iP2T^^H0[;[V10000000202T_^H0[<;^0:BniP2/`^h0Y;kV0000002T^^H0[;kV0`0000001P2/`^h0
Y<;^0:c2kP2T`^h0[<;^0:C2kP<0000000<0Y<K^0:C6mP0000001P2T`^h4000000<0Y<K^00D00000
0:C6kP000000Y<K^0:C2kP03000000H0Y<K^0`0000000`2Ta^h0Y<;^000000070?ooo`030000003o
ool0oooo00H0oooo000;0?ooo`D000000`3oool00`000000oooo0000000200000080oooo00H00000
0:b1Y02/PJ`0[86T0:b1[00000030=koc@8000004@3oool0=@000000W76L0:AaW02LLI`0Y76L09aa
W02TLI`0W76L0:AaW02LLI`0Y76L09aaW02TLI`0W76L0:AaW0000000W76T09a]W0000000[86T0:b5
[02/PJ@0[8F/0:b1Y02/QJ`0[86T0:b5[02/PJ@0[8F/0000002TPJ`0Y7f/0:B1[02TOJ`000000;BM
_@2mWLD0]9fm0;fMa@2dWKd0_Ig50;BM_@2mWLD0]9fm0;fMa@000000aLKN0;g2gP35a]h0_L;N0<G6
gP2m`]h0aLKN00D000001@2DNK@00`000000U97E08^Ac@040000000A09b1_@2LPK@0W86m09b1]02L
PKd0W86d0000002LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd00P0000003`2L
SKd0000009bIc@2TVLd0W9W=0:BIc@2LVLd0Y9W=09bIc@2TVLd0W9W=0:BIc@2LVLd0Y9W=09bIc@03
0000000<0:BQc@2TXMD0Y:7=0:BQe@2TXLd0Y:7E0:BQc@2TXMD0Y:7=0:BQe@2TXLd0Y:7E0`000000
2P2/Z]h0Y:[E0:bZgP2TZ]D0[:[N0:BZe@2/Z]h0Y:[E0:bZgP2TZ]D20000000;0:bfgP2//]h00000
0:BbgP2/]^H0Y;;N0:bfiP2T/]h0[;KV0:BbgP2/]^H00`0000001@2/_^H0[;[V0:bniP2/^^H00000
0080[;[V0`0000090:c2kP<0000000D0[<K^0:C2kP2/a^h0Y<;^0:c6kP04000000090:C6kP2/b_H0
Y<K^0:c:mP2Ta^h0[<[f0000002/a^h0Y<;^00<0000000P0Y<K^0:c6mP2Ta^h0[<Kf0:C6kP2/a_H0
Y<K^0:c6mP@0000000D0Y<K^0:c6mP2Ta^h0[<Kf0:C6kP04000000060:c6mP2Ta^h0[<Kf0:C6kP2/
a_H000001`3oool00`000000oooo0?ooo`060?ooo`003@3oool00`000000oooo0?ooo`030?ooo`80
00001@3oool02@000000[86T0:b1Y02/PJ@000000=koc@3Eold0g_o=0000000C0?ooo`0P0000002T
LI`0W6fL0:AaW02LKI`0Y76L09a]W02TLI`0W6fL0:AaW02LKI`0Y76L09a]W02TLI`0W6fL0000002D
KI`000000:B1Y02/PJ`0Y86T0:b1[02TPJ@0[86/0:B1Y02/PJ`0Y86T0:b1[02TPJ@0[86/0000002T
PJ`2000000X0]9fm00L000000;goiP000000aL;N0;g2e@35`]h0_L;E00<0000000P0RgFd09Ai]02;
MK@0U7Vd08]e]02DNK@0RgFd09Ai]0@0000001<0U86d09b1_@2DPK@0W86m09B1]02LPKd0U86d09b1
_@2DPK@0W86m0000002LTLd0U97509bAc@2DTLD0W97=09BAa@2LTLd0U9750080000000l0Y8fm09b9
_@2TSKd0W8Vm0000002LULd0Y9W=09bEc@2TVLd0W9G=0:BIc@2LULd0Y9W=09bEc@2TVLd00P000000
3@2TULD000000:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd00P000000
2`2/YMD0Y:G=0000002TZ]D0[:[E0:BZe@2/Z]D0Y:[E0:bZe@2TZ]D0[:[E00800000102//]h01P00
0000[;;N0:BbgP2//]h0Y;;N0:bbgP<0000000L0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV00<0
000000T0[<;^0:BniP2/`^h0Y;kV0:c2kP2T_^H0[<;^0:BniP2/`^h00`0000001@2/b_H0Y<K^0000
002T`^h0[<;^00<0000000X0Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0`00
00070:C6kP@0000000@0Y<Kf0000002Ta_H000000P2Ta^h3000000080:C2kP2Ta^h0Y<;^0:C6kP2T
`^h0Y<K^0:C2kP2Ta^h3000000H0oooo00<000000?ooo`3oool01P3oool000d0oooo00<000000?oo
o`3oool0103oool00`000000oooo0?ooo`040?ooo`050000002/PJ`0[86T0000003Nold00P00000D
0?ooo`0?0000002LLI`0Y76L09aaW02TLI`0W76L0:AaW02LLI`0Y76L09aaW02TLI`0W76L0:AaW02L
LI`0Y76L0080000000d0[86T0:b5[02/PJ@0[8F/0:b1Y02/QJ`0[86T0:b5[02/PJ@0[8F/0:b1Y02/
QJ`0[86T0080000000X0_Ig50;BM_@2mWLD0]9fm0;fMa@2dWKd0_Ig50;BM_@2mWLD0]9fm0P000000
1@2monH0_O[V0;goiP000000aLKN008000002@2DNK@20000000D09ai[0000000W86m09b1]02LPKd0
W86d09b1_@2LPK@0W86m09b1]02LPKd0W86d09b1_@2LPK@0W86m0000002LULd0W97=09bEc@2LTLd3
0000000?09b=_@2TSLD0W8fm0:B=a@2LSKd0Y8g50000002TVLd0W9W=0:BIc@2LVLd0Y9W=09bIc@2T
VLd0W9W=0080000000`0[9W=0:BIa@2/VLd0Y9W50000002TXLd0Y:7E0:BQc@2TXMD0Y:7=0:BQe@2T
XLd3000000@0[:GE00H000000:bZgP2TZ]D0[:[N0:BZe@2/Z]h30000000:0:bfgP2//]h0[;KN0:bb
gP2/]]h0[;;N0:bfgP000000[;KV0:BbgP80000000P0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV
0:bjiP80000000<0[<[f0:c:kP0000001@2/`^h4000000050:c:mP2Tb^h0[<[f0:C:kP2/b_H00`00
00002@2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0Y<K^0:c:mP04000000060:C6kP2Tb_H00000
0:c6mP2Ta^h0[<Kf100000001`2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0000002Ta^h00`0000001`2Ta^h0
[<Kf0:C6kP2/a_H0Y<K^0:c6mP2Ta^h010000000102/`^h0Y<;^0:c2kP0000050?ooo`030000003o
ool0oooo00H0oooo000=0?ooo`030000003oool0oooo00D0oooo0P0000040?ooo`050000002/PJ@0
00000=Goc@0000005P3oool07`000000Y76L09a]W02TLI`0W6fL0:AaW02LKI`0Y76L09a]W02TLI`0
W6fL0:AaW02LKI`0Y76L09a]W0000000Y86T0:b1[02TPJ@0[86/0:B1Y02/PJ`0Y86T0:b1[02TPJ@0
[86/0:B1Y02/PJ`0Y86T0:b1[00000002P2dWKd01P000000]?[N0;goiP2dn]h0_OoV0;CjgP800000
00T0U6fT0000002DNK@0RgFd09Ai]02;MK@0U7Vd08]e]02DNK@00P000000502LMJ@0W7V/09aeY02L
NJ`0000009b1_@2DPK@0W86m09B1]02LPKd0U86d09b1_@2DPK@0W86m09B1]02LPKd0U86d0000002D
TLD0W97=0P0000003`2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd000000:BI
c@2LULd0Y9W=09bEc@030000000;0:BEa@2TVLd0Y9G50:BIc@2TULD0Y9W=0000002TXMD0Y9g=0:BQ
e@2TWLd00`0000002`2TYLd0[:GE0:BUc@2/YMD0Y:G=0:bUe@2TYLd0[:GE0000002/Z]D0Y:[E0080
00002P2//]h3000000070:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP03000000070:C6kP2/b^h0
Y<K^0:c:kP000000[<;^0:BniP03000000060:C6kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f0`0000002@2T
c_H0Y<[f0000002Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP03000000060:C:mP2Ta^h0Y<[f0:C6kP2T
b_H0Y<K^10000000202Ta_H0Y<K^0:C6mP2Ta^h0Y<Kf0:C6kP2Ta_H0Y<K^0`0000020:C2kP040000
002Ta^h0Y<;^0:C6kP@000001`2T`^h2000000D0oooo00<000000?ooo`3oool01@3oool000d0oooo
00<000000?ooo`3oool01`3oool010000000oooo0?ooo`3oool4000001L0oooo00h0000009aaW02T
LI`0W76L0:AaW02LLI`0Y76L09aaW02TLI`0W76L0:AaW02LLI`0Y76L09aaW080000000d0[8F/0:b1
Y02/QJ`0[86T0:b5[02/PJ@0[8F/0:b1Y02/QJ`0[86T0:b5[02/PJ@0[8F/0080000000h0]9fm0;fM
a@2dWKd0_Ig50;BM_@2mWLD0]9fm0;fMa@2dWKd000000;gjiP2monH0_O[V0;goiP800000102DKJ@0
10000000U7Vd09Ai]02DNK@3000000H0W7V/00d0000009b1]02LPKd0W86d09b1_@2LPK@0W86m09b1
]02LPKd0W86d09b1_@2LPK@0W86m00<0000000l0Y8g509b=_@2TSLD0W8fm0:B=a@2LSKd0Y8g509b=
_@2TSLD0W8fm0:B=a@2LSKd0000009bIc@2TVLd00P000000302TVLD0[9W=0:BIa@2/VLd0Y9W50:bI
c@2TVLD0[9W=0:BIa@2/VLd000000:BQe@8000002`2/YMD3000000090:bbgP2/]]h0[;;N0:bfgP2/
/]h0[;KN0:bbgP2/]]h0[;;N00<0000000L0[<;V0;C6kP000000[;kV0:bjiP2/_^H0[;[V00<00000
00L0[<[^0:c:mP2/b^h0[<[f0:c:kP2/b_H0[<[^00<0000000D0Y<[^0:c:mP2Tb^h0[<[f0:C:kP04
000000D0Y<kf00<000000:c:mP2Ta^h0100000001@2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f00@0000000P0
Y<K^0:C6mP000000Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP@0000000D0Y<;^0:C6kP2T`^h000000:C2kP04
000000080:c2kP2T`^h0[<;^0:C2kP2/`^h0Y<;^0:c2kP2T`^h3000000030:BniP2/_^h0000000@0
oooo00<000000?ooo`3oool01@3oool000`0oooo0P00000:0?ooo`030000003oool0oooo00800000
6@3oool080000000Y76L09a]W02TLI`0W6fL0:AaW02LKI`0Y76L09a]W02TLI`0W6fL0:AaW02LKI`0
00000;BA[0000000Y86T0:b1[02TPJ@0[86/0:B1Y02/PJ`0Y86T0:b1[02TPJ@0[86/0:B1Y02/PJ`0
00000=G:e@3=b/d000001P2dWKd2000000030;CjgP2monH0]?[N008000001P2DKJ@060000000RgFd
000000000000W7V/09aeY02LNJ`0W7FT09ai[02LMJ@0W7V/09aeY02LNJ`0000009b1_@2DPK@0W86m
09B1]02LPKd0U86d09b1_@2DPK@0W86m09B1]080000000l0Y8Fd0000002TSKd0W8Vm0:B=_@2LRKd0
Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd00`0000002`2TVLd0Y9G50:BIc@2TULD0
Y9W=0:BEa@2TVLd0Y9G50:BIc@2TULD0Y9W=00<0000000/0[:GE0:BUc@2/YMD0Y:G=0:bUe@2TYLd0
[:GE0:BUc@2/YMD0Y:G=0:bUe@02000000030:bfgP2/^]h0000000D0[;;N10000000202/`^H0[<K^
0:c2iP2/a^h0[<;V0000002/^^H0[;kV0P0000001`2Ta^h0[<[^0:C6kP2/b^h0Y<K^0:c:kP2Ta^h0
0`0000020:CBmP030000002/b_H0Y<K^00<0000000P0Y<[f0:C>mP2Tb_H0Y<kf0:C:mP2Tc_H0Y<[f
0:C>mP@0000000H0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP2Tb_H3000000090:C6mP2T`^h0Y<Kf0:C2kP2T
a_H0Y<;^0000002Ta^h0Y<Kf00<000001P2T`^h3000000030:BniP2T_^h0000000D0Y<;^10000000
1P2T_^H0Y;k^0:BniP2T_^h0Y;kV000000@0oooo00<000000?ooo`3oool01@3oool001T0oooo00<0
00000?ooo`3oool06P3oool0:0000000W76L0:AaW02LLI`0Y76L09aaW02TLI`0W76L0:AaW02LLI`0
Y76L09aaW02TLI`000000;fE]0000000[8F/0:b1Y02/QJ`0[86T0:b5[02/PJ@0[8F/0:b1Y02/QJ`0
[86T0:b5[0000000eL[E0=G:c@3Eb]D000000;fMa@2dWKd0_Ig50;BM_@2mWLD000000;goiP2mn^H3
000000P0U6fT0P00000<09ai[0080000002LPKd0W86d09b1_@2LPK@0W86m09b1]02LPKd2000000@0
Y8Fd00/000000:B=a@2LSKd0Y8g509b=_@2TSLD0W8fm0:B=a@2LSKd0Y8g509b=_@020000000<0:bE
a@000000[9W=0:BIa@2/VLd0Y9W50:bIc@2TVLD0[9W=0:BIa@2/VLd0Y9W50P0000000`2dZ]D0[:G=
000000080:bUe@<0000000L0];[V0:bfgP2d^^H0[;KN0000002//]h0[;KN00<0000000T0]<K^0:c2
iP2da^h0[<;V0;C6kP2/`^H0]<K^0:c2iP2da^h00`0000001@2/b_H0[<[^0:c:mP2/b^h0[<[f00@0
000000D0Y=;f0:CFo`2Td_H0Y=Ko0:CBmP03000000L0Y<kf100000030:C:mP030000002Tb_H0Y<K^
00@0000000P0Y<Kf0:C6kP2Ta_H0Y<K^0:C6mP2Ta^h0Y<Kf0:C6kP@0000000D0Y<K^0:C2kP2Ta^h0
Y<;^0:C6kP04000000080:BnkP2/`^h000000:c2kP2T_^h000000:C2kP2/`^h3000000090:bnkP2T
_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^008000000`3oool00`000000oooo0?ooo`050?oo
o`006P3oool2000001X0oooo024000000:AaW02LKI`0Y76L09a]W02TLI`0W6fL0:AaW02LKI`0Y76L
09a]W02TLI`000000;fA[02dTJ`0_I6/0000002/PJ`0Y86T0:b1[02TPJ@0[86/0:B1Y02/PJ`0Y86T
0:b1[0000000eL[E0<g:c@3Eb]D0cL[=0=G:e@0000000`2dWKd010000000_OoV00000000000909A]
Y0<000000180W7FT09ai[02LMJ@0W7V/09aeY02LNJ`0W7FT09ai[02LMJ@0W7V/09aeY02LNJ`0W7FT
0000002DPK@0W86m09B1]02LPKd30000000>0:B5]02LQK@0Y8Fd09b5]02TQK@0W8Fd0000002LRKd0
Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd30000000<0:BE_@2/ULD0Y9Fm0:bEa@000000Y9W=0:BEa@2T
VLd0Y9G50:BIc@2TULD0Y9W=0P0000002P2/YLd0[:[E0:bUc@2/Z]D000000:bUe@2TYLd0[:GE0:BU
c@2/YMD3000000070:bjgP2/]]h0[;[N0:bfgP2/^]h0[;KN0:bjgP03000000090:c6kP2/`^H0[<K^
0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^00<000000P2Te_H00`000000Y<K^0:c:kP03000000H0Y=;f
0`0000001P2Ld_H0W=;o0000002Tc_H0Y<[f0:C>mP@0000000L0W<[^0:C:mP2Lb^h0Y<[f09c:kP2T
b_H0W<[^00<0000000P0Y<;^0:C6mP2T`^h0Y<Kf0:C2kP2Ta_H0Y<;^0:C6mP@0000000@0Y;kV0:Bn
kP2T_^H000000P2T`^h3000000090:BnkP2T_^H0Y;k^0:BniP2T_^h0Y;kV0:BnkP000000Y;k^00<0
000000P0Y;k^0:BniP2T_^h0Y;kV0:BnkP2T_^H0Y;k^0:BniP@0000000<0[;[V0:BjiP0000000P3o
ool00`000000oooo0?ooo`050?ooo`00703oool00`000000oooo0?ooo`0G0?ooo`0R0000002LLI`0
Y76L09aaW02TLI`0W76L0:AaW02LLI`0Y76L09aaW02TLI`000000;fE]02mTJ`0_IFd0;fA[0000000
[86T0:b5[02/PJ@0[8F/0:b1Y02/QJ`0[86T0:b5[0000000eL[E0=G:c@3Eb]D0eL[=0=G:e@3Eb/d0
00000;BM_@D000002@2DKJ@010000000W7FT0:AeY000000>09ai[0040000002LPK@00000000000T0
Y8Fd00D000000:B=a@2LSKd0Y8g509b=_@03000000L0[9G500D000000:BIa@2/VLd0Y9W50:bIc@03
000000080;BZe@2/YLd0]:[E0:bUc@2dZ]D0[:G=0;BZe@0000020:bUe@80000000P0];[V0:bfgP2d
^^H0[;KN0;BjiP2/]]h0];[V0:bfgP8000000P2/e_H01P000000[<;V0;C6kP2/`^H0]<K^0:c2iP@0
000000D0[=[o0:CFmP2/f_l0Y=Kf0:cJo`03000000050:CFo`2Td_H0Y=Ko0:CBmP2Te_l010000000
1@2Ld_H0Y=;o09cBmP2Td_l0W=;f00@000001`2Tb_H4000000060:C2kP2T`_H000000:C6mP2Ta^h0
Y<Kf1@0000001`2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h00`000000202/`^h0Y;k^0:c2kP2T
_^h0[<;^0:BnkP2/`^h0Y;k^10000000202/_^h000000:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0`00
00001`2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP0000000P3oool00`000000oooo0?ooo`050?ooo`00
7@3oool00`000000oooo0?ooo`0F0?ooo`0G0000002TLI`0W6fL0:AaW02LKI`0Y76L09a]W02TLI`0
W6fL0:AaW0000000_I6/0;BA[02mTJ`0]96/0;fA[0000000[86/0:B1Y02/PJ`0Y86T0:b1[02TPJ@0
0P000000203Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0P000000102LJI`0U6VD09aYW000
000709A]Y08000000180W7FL09aeY02LMI`0W7FT0000002LNJ`0W7FT09ai[02LMJ@0W7V/09aeY02L
NJ`0W7FT09ai[02LMJ@0W7V/09aeY02LNJ`20000000>09b5]02TQK@0W8Fd0:B5]02LQK@0Y8Fd09b5
]02TQK@0W8Fd0:B5]02LQK@0Y8Fd0000002TSKd20000000=0:BE_@2/ULD0Y9Fm0:bEa@2TUKd0[9G5
0:BE_@2/ULD0Y9Fm0:bEa@2TUKd000000:BEa@020000000:0:bZe@2/YLd0[:[E0:bUc@2/Z]D0[:G=
0:bZe@2/YLd0[:[E0:bUc@<0000000L0[;KN0:bjgP2/]]h0[;[N0:bfgP2/^]h0[;KN00<0000000L0
[=;f0:cFmP2/d_H0[=Kf0000002/a^h0[<;V00<000001@2Te_H40000000309cNo`2Df_l000000080
Y=;f0`0000001@2Ld_l0W=;f09cBo`2Ld_H0W=;o00@0000000L0W<Kf09c:mP000000000009c:kP2T
b_H0W<[^00@0000000H0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h4000000090:BnkP2T_^H0Y;k^0:Bn
iP2T_^h0Y;kV0:BnkP2T_^H0Y;k^00<0000000P0Y;[V0000002T_^h0Y;kV0:BnkP2T_^H0Y;k^0:Bn
iP<0000000P0Y;[V0:bniP2T^^H000000:BjiP2/_^H000000:BnkP@0000000P0Y;[V0:bjiP2T^^H0
[;[V0:BjiP2/^^H0Y;[V0:bjiP<0000000<0oooo0000003oool01P3oool001h0oooo00<000000?oo
o`3oool05@3oool07`000000W76L0:AaW02LLI`0Y76L09aaW02TLI`0W76L0:AaW02LLI`000000;fA
[02mUK@0_I6/0;fE]02mTJ`000000:b1Y02/QJ`0[86T0:b5[02/PJ@000000=G:c@3Eb]D0eL[=0=G:
e@3Eb/d0eL[E0=G:c@3Eb]D00P0000001P2DJI`0W6VL09AYW02LJI`0U6VL000000H0U6fT00P00000
0:AeY02LMJ@0Y7FT09aeY02TMJ@0W7FT000000`0W7V/0P00000=0:B5]08000003@2/ULD20000000:
0:bUc@2dZ]D0[:G=0;BZe@2/YLd0]:[E0:bUc@2dZ]D0[:G=0;BZe@80000000L0]<[V0;C:kP000000
];[V0:bfgP2d^^H0[;KN00<000001`2/e_H3000000040:CFmP2/f_l0Y=Kf0:cJo`@000001P2Lg_l3
000000040:CBo`2Ld_H0Y=;o09cBmP@0000000H0Y<[f09c:mP2Tb_H0W<[f0:C:mP2Lb_H500000007
0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP03000000090:BjiP2T^^h000000:C2kP2T_^h0Y<;^
0:BnkP2T`^h0Y;k^00@0000000L0[;k^0:BjiP2/_^h0Y;[V0:bnkP000000[<;^00@0000000L0Y;[V
0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V00@0000000P0[;k^0:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^
0:BjiP@0000000D0[;k^0:BjiP000000oooo000000070?ooo`007`3oool2000001H0oooo02L00000
09a]W02TLI`0W6fL0:AaW02LKI`0Y76L09a]W0000000]96/0;fA[02dTJ`0_I6/0;BA[02mTJ`00000
0:b1[02TPJ@0[86/0:B1Y0000000cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@000000W6VL
09AYU02LJI`0U6VD09aYW02DJI@0W6VL09AYU00000000`2DKJ@20000000B09aeY02LMI`0W7FT09ae
W02LMJ@0W7FL09aeY0000000W7V/09aeY02LNJ`0W7FT09ai[02LMJ@0W7V/09aeY02LNJ`0W7FT0P00
00003@2/QJ`000000:B5]02LQK@0Y8Fd09b5]02TQK@0W8Fd0:B5]02LQK@0Y8Fd09b5]02TQK@00`00
0000302/ULD0Y9Fm0:bEa@2TUKd0[9G50:BE_@2/ULD0Y9Fm0:bEa@2TUKd0[9G50:BE_@80000000X0
];;E0000002/Z]D0[:G=0:bZe@2/YLd0[:[E0:bUc@2/Z]D0[:G=0P0000001`2db^h0]<[V0;C:kP2d
b^H000000:bfgP2/^]h00P0000001`2/d_H0[=Kf0:cBmP2/e_H0[=;f0:cFmP2/d_H00`0000000`2;
h_H0U>Ko08_RmP040000000609cNo`2Df_l0W=ko09CJo`2Lg_l0U=[o100000000`2Le_l0U=;o0000
00040000000609c6mP2Lb_H0W<Kf09c:mP2La_H0W<[f100000002@2L^^H0Y;k^09bjiP000000W<;^
0:C2kP2L`^h0Y<;^09c2kP03000000H0Y;[V00<000000:BnkP2T_^H00`0000080:BjiP<0000000T0
[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0100000001@2T^^H000000:BjiP2/^^H0
Y;[V00D0000000H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H2000000030?ooo`000000oooo00D0oooo
000Q0?ooo`030000003oool0oooo01<0oooo01/000000:AaW02LLI`0Y76L09aaW02TLI`0W76L0000
002mTJ`0_IFd0;fA[02mUK@0_I6/0;fE]02mTJ`000000:b1Y02/QJ`0[86T0000003Eb/d0eL[E0=G:
c@3Eb]D0eL[=0=G:e@3Eb/d00P0000002P2LJI`0U6VL09aYW02DJI`0W6VL09AYW02LJI`0U6VL09aY
W000000209A]Y00;0000002LMJ@0Y7FT09aeY02TMJ@0W7FT0:AeY02LMJ@0Y7FT09aeY00000002@2L
NJ`01P000000Y8F/0:b5]02TQJ`0[8Fd000000T0Y8Fd00D000000;BIa@2/VKd0]9W5000000090:bE
a@80000000T0];;E0;ffe@2d/]D000000:bUc@2dZ]D0[:G=0;BZe@2/YLd00`0000001P2db^H0]<[^
0;C:iP2db^h0]<[V0;C:kP<000001P2/e_H3000000<0U>Ko0`0000000`1kg_H0Pmko0000000309cN
o`@0000000<0U=Ko09cFo`0000000`0000001`2L`_H0Y<Kf09c2mP000000W<[f0:C:mP2Lb_H01000
00002P2T_^h0Y;[^0:BnkP2T^^h0Y;k^0:BjkP2T_^h000000:C2mP2T`^h3000000080:BjiP2T^^h0
Y;[V0:BjkP2T^^H0Y;[^0:BjiP2T^^h4000000080:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/
_^h3000000080:bjiP2T^^H000000:BjiP2/_^h0Y;[V0:bnkP2T^^H4000000040:bniP2T^^H0[;kV
0:BjiPD0000000P0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V0:bnkP<0000000@0[;k^0:BniP00
000000001P3oool00280oooo00<000000?ooo`3oool04P3oool09P000000W6fL0:AaW02LKI`0Y76L
09a]W0000000]96/0;fA[02dTJ`0_I6/0;BA[02mTJ`0]96/0;fA[02dTJ`000000:B1Y0000000cL[=
0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0000002DJI@0W6VL09AYU02LJI`0U6VD09aYW02DJI@0W6VL
09AYU02LJI`0U6VD0`0000008@2LMI`0W7FT09aeW02LMJ@0W7FL09aeY02LMI`0W7FT09aeW02LMJ@0
W7FL0000002LMJ@0W7V/09aeY02LNJ`0W7FT09ai[02LMJ@000000:B1[02/QJ`0Y86/0:b5[02TPJ`0
000009b5]02TQK@0W8Fd0:B5]02LQK@0Y8Fd09b5]0020000000>0;BIa@2/VKd0]9W50:bI_@000000
Y9Fm0:bEa@2TUKd0[9G50:BE_@2/ULD0Y9Fm0:bEa@0000060;Bbe@030000002/YLd0[:[E00800000
00L0]<[V0;C:kP2db^H0]<[^0;C:iP2db^h0]<[V008000000P2Dj_H2000000030:cFmP2/d_H00000
0080000000<0Rn;f09CVo`0000000`000000101kg_H0Nmko07_NmP1kg_l50000000309cFo`2Dd_l0
000000<0000000H0W<;f09c2kP2L`_H0W<;^09c2mP2L`^h50000000:09bjiP2T_^h0W;[V0:BnkP2L
^^H0Y;k^09bjiP2T_^h0W;[V0:BnkP@000001`2T^^H4000000040:BfiP2T/]h0Y;KV000000D0Y;[V
0`000000202/^^H0Y;KV0:bjiP2T]^H0[;[V0000002/_^H0Y;[V0`0000001P2T^^H0[;[V0:BjiP2/
^^H0Y;[V0:bjiP<0000000P0[;kV0:BjiP000000Y;[V0:bjiP2T^^H0[;[V0:BjiP@0000000D0[;k^
0:BniP2/_^h0Y;kV0:bnkP02000000H0oooo000S0?ooo`030000003oool0oooo0140oooo01000000
0:AaW02LLI`0Y76L09aaW0000000_I6/0;fE]02mTJ`0_IFd0;fA[02mUK@0_I6/0;fE]02mTJ`0_IFd
0P0000008`3Eb/d0eL[E0=G:c@3Eb]D0eL[=0=G:e@3Eb/d0000009AYW02LJI`0U6VL09aYW02DJI`0
W6VL09AYW02LJI`0U6VL09aYW02DJI`0W6VL0000002TMJ@0W7FT0:AeY02LMJ@0Y7FT09aeY02TMJ@0
W7FT0:AeY02LMJ@0Y7FT09aeY02TMJ@0000000D0W7V/0P0000001`2TQJ`0[8Fd0:B5[02/QK@0Y8F/
0:b5]00000001P2TQK@02@000000[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm000000050:bEa@80
000000L0_KKE0;Bbe@2m]]D0];;E0;ffe@2d/]D0_KKE00<0000000L0]<[^0;C:iP2db^h0]<[V0;C:
kP2db^H0]<[^0080000000D0U>[f09C^o`2Dj_H0U>ko09CZmP0300000080U>Ko0`0000000`23g_l0
Nmkf000000050000000309CBo`2;d_l0U=;o00@0000000H0W<;f0:C6mP2L`_H0Y<Kf09c2mP2Ta_H4
000000<0Y;;V00P000000:BjkP2T_^h0Y;[^0:BnkP2T^^h0Y;k^0:BjkP@0000000L0Y;;V0:BbgP2T
/^H000000:BjkP2T^^H0Y;[^00@0000000X0Y;KV0:bfiP2T]^H0[;KV0:BfiP2/]^H0Y;KV0000002T
^^H0[;k^0`000000202/^^H0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V0`0000001@2T^^H0[;kV
0:BjiP2/_^H0Y;[V00@0000000P0[;k^0:BjiP2/_^h000000:bnkP000000[;k^0:BjiP<0000000T0
Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP2T_^H00P0000060?ooo`00903oool200000140
oooo01L0000009a]W02TLI`0W6fL0:AaW0000000_I6/0;BA[02mTJ`0]96/0;fA[02dTJ`0_I6/0;BA
[02mTJ`0]96/0000003=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@030000000;09aYW02DJI@0W6VL09AY
U02LJI`0U6VD09aYW02DJI@0W6VL09AYU02LJI`00P000000802LMI`0W7FT09aeW02LMJ@0W7FL09ae
Y02LMI`0W7FT09aeW02LMJ@0W7FL09aeY02LMI`0000009aeY02LNJ`0W7FT09ai[0000000[8F/0:B1
[02/QJ`0Y86/0:b5[02TPJ`0[8F/0:B1[0000000W8Fd0:B5]02LQK@0Y8Fd0P000000302/VKd0]9W5
0:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50000002/ULD0Y9Fm0:bEa@8000001`2d/]D2000000070:c^
mP2Tk_H000000;C:iP2db^h0]<[V0;C:kP03000000<0U>[f100000000`11`/D00000000000070000
000508_Bo`2Dd_l0Rm;o09CBo`2;d_l0100000001P2L]^h0W;KV0000002L`^h0W<;f09c2kP@00000
00/0Y;;V09bbgP2T/^H0W;;N0:BbiP2L/]h0Y;;V0000002T_^h0W;[V0:BnkP04000000070:BbgP2T
/^H0Y;;N0:BbiP2T/]h0Y;;V0:BbgP040000000:0:BfiP2T/]h0Y;KV0:BbgP2T]^H0Y;;N0:BfiP2T
/]h0Y;KV0:BbgP@0000000L0[;[V0:BfiP2/^^H0Y;KV0:bjiP2T]^H0[;[V00@0000000D0Y;[V0:bj
iP000000[;[V0:BjiP03000000080:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H400000009
0:BniP2/_^h0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV00<0000000<0oooo0000003oool01@3o
ool002H0oooo00<000000?ooo`3oool03P3oool03`000000Y76L09aaW02TLI`000000;fE]02mTJ`0
_IFd0;fA[02mUK@0_I6/0;fE]02mTJ`0_IFd0;fA[003000000050=G:c@3Eb]D0eL[=0=G:e@000000
0P3oool070000000U6VL09aYW02DJI`0W6VL09AYW02LJI`0U6VL09aYW02DJI`0W6VL0000002/NJ@0
00000:AeY02LMJ@0Y7FT09aeY02TMJ@0W7FT0:AeY02LMJ@0Y7FT09aeY02TMJ@0W7FT0:AeY0000003
09ai[00<0000002/QK@0Y8F/0:b5]02TQJ`0[8Fd0:B5[02/QK@0Y8F/0:b5]02TQJ`000000P2TQK@0
30000000]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@<0000000L0_KKE
0;Bbe@2m]]D0];;E0;ffe@2d/]D0_KKE0080000000L0[>kf0:cbo`2/k_H0[?;o0000002db^h0]<[V
0080000000<0U>ko09CZmP0000002P000000202;`_H0U<Ko08_2mP2Da_l0000009CBo`2;d_l0U=;o
100000001P2L]^H0W;K^09bfiP2L]^h0W;KV09bfkP@000002`2T/^H4000000080:BbiP2T/]h0Y;;V
0:BbgP2T/^H0Y;;N0:BbiP2T/]h30000000:0:B^gP2//]h000000:bfiP2T]^H0[;KV0:BfiP2/]^H0
Y;KV0:bfiP@0000000L0Y;KN0:bfiP2T]]h000000:BjiP2/^^H0Y;[V00@0000000H0[;[V0:BjiP2/
^^H0Y;[V0:bjiP2T^^H3000000090:BjiP2/_^h0Y;[V0:bnkP2T^^H0[;k^0:BjiP2/_^h0Y;[V00@0
000000L0[<;^0000002/_^h0Y;kV0:bnkP2T_^H0[;k^00@00000103oool00`000000oooo0?ooo`04
0?ooo`009`3oool00`000000oooo0?ooo`0=0?ooo`0>0000002LKI`0Y76L0000002mTJ`0]96/0;fA
[02dTJ`0_I6/0;BA[02mTJ`0]96/0;fA[02dTJ`5000000030<g:c@000000000000<0oooo01`00000
09aYW02DJI@0W6VL09AYU02LJI`0U6VD09aYW02DJI@0W6VL0000002TNJ@0Y7FL0000002LMI`0W7FT
09aeW02LMJ@0W7FL09aeY02LMI`0W7FT09aeW02LMJ@0W7FL09aeY02LMI`0W7FT0`0000002`2/QJ`0
Y86/0:b5[02TPJ`0[8F/0:B1[02/QJ`0Y86/0:b5[02TPJ`0[8F/00<0000000d0]9W50:bI_@2dVLD0
[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0000000L0];;E0P0000001P2/k_H0Y>kf
0:c^mP2Tk_H0[>kf0:C^mP`0000000T0U<Ko08_2mP2Da_l0Rl;f09C6o`2;`_H0U<Ko08_2mP2Da_l0
1@0000001P2L]^h0W;KV09bfkP2L]^H0W;K^09bfiP@0000000/0Y:GE09bQe@000000W;;N0:BbiP2L
/]h0Y;;V09bbgP2T/^H0W;;N0:BbiP04000000080:BUe@2TZ]h000000:BbiP2T/]h0Y;;V0:BbgP2T
/^H4000000090:B^gP2//]h0Y:kN0:bbgP2T[]h000000:BbgP2T]^H0Y;;N00@0000000L0[;KV0:Bf
gP2/]^H0Y;KN0:bfiP2T]]h0[;KV00@0000000L0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/^^H0Y;[V00<0
000000P0Y;kV0:bnkP000000[;kV0:BjiP2/_^H0Y;[V0:bniP@0000000P0Y;kV0:c2kP2T_^H00000
0:BniP000000Y;kV0:bnkP<00000203oool00`000000oooo0?ooo`040?ooo`00:03oool00`000000
oooo0?ooo`0<0?ooo`0=0000002TLI`000000;fE]02mTJ`0_IFd0;fA[02mUK@0_I6/0;fE]02mTJ`0
_IFd0;fA[007000000H0oooo03H0000009aYW02DJI`0W6VL09AYW02LJI`0U6VL09aYW02DJI`00000
0:AiW02/NJ@000000:AeY02LMJ@0Y7FT09aeY02TMJ@0W7FT0:AeY02LMJ@0Y7FT09aeY02TMJ@0W7FT
0:AeY02LMJ@000000:B5[02/QK@0Y8F/0:b5]02TQJ`0[8Fd0:B5[02/QK@0Y8F/0:b5]02TQJ`0[8Fd
0:B5[0000000[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD20000
00060<GJiP000000];;E0;ffe@2d/]D0_KKE0P0000001@2/k_H0[?;o0:c^mP2/l_l0[>kf00H00000
00H0U:[V09BUiP2DZ^H0U:GV09BZiP2DYNH20000000409C6o`2;`_H0U<Ko08_2mPH0000000L0Y:GN
09bQe@2TYMh0000009bfkP2L]^H0W;K^00@0000000P0Y:7E0:BUgP2TXMD0Y:GN0:BQe@2TYMh0Y:7E
000000<0Y;;V100000002@2/Z]h0Y:[E0:bZgP2TZ]D0[:[N0:BZe@000000Y;;N0:BbiP0300000009
0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N00@0000000P0Y;KN0:bfiP2T]]h0[;KV
0:BfgP2/]^H0Y;KN0:bfiP<0000000P0[;kV0:bjiP000000Y;[V0:bjiP2T^^H0[;[V0:BjiP<00000
00P0Y;kV0:c2kP2T_^H0[<;^0:BniP000000Y;[V0:bnkP<0000000P0[<;^0:BnkP2/`^h0Y;k^0:c2
kP2T_^h0[<;^0:BnkP@00000303oool00`000000oooo0?ooo`030?ooo`00:@3oool2000000`0oooo
00d0000009a]W0000000]96/0;fA[02dTJ`0_I6/0;BA[02mTJ`0]96/0;fA[02dTJ`0_I6/00H00000
1`3oool06P000000U6VD09aYW02DJI@0W6VL09AYU02LJI`0U6VD0000002TMI`0Y7VT0:AeW02TNJ@0
000009aeY02LMI`0W7FT09aeW02LMJ@0W7FL09aeY02LMI`0W7FT09aeW02LMJ@0W7FL0P000000302/
QJ`0Y86/0:b5[02TPJ`0[8F/0:B1[02/QJ`0Y86/0:b5[02TPJ`0[8F/0:B1[080000000X0]9W50:bI
_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0P0000001035f^h0_MKV0<GJkP0000020;Bb
e@80000000@0[>kf0:C^mP2/k_H0Y>kf0`000000302DRLD0000009BUgP2DZ^H0U:GN09BZiP2DYMh0
U:[V09BUgP2DZ^H0U:GN09BZiPH000002@2LXMD40000000;0:BUe@2LXMD0Y:GE09bQe@2TYMD0W:7E
0:BUe@2LXMD0Y:GE09bQe@2TYMD0100000002P2TYMD0Y:[N0:BUe@2TZ]h0Y:GE0:BZgP2TYMD0Y:[N
0:BUe@2TZ]h3000000090:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N00<0000000P0
[;KN0:BbgP000000Y;KN0:bfiP2T]]h0[;KV0:BfgP@0000000P0[;[V0:BjiP2/^^H0Y;[V0:bjiP00
0000[;[V0:BjiP<0000000P0Y;kV0:bnkP2T_^H0[;k^0:BniP2/_^h0Y;kV0:bnkP<0000000T0[<;^
0:BniP2/`^h0Y;kV0:c2kP2T_^H0[<;^0:BniP2/`^h01000000>0?ooo`030000003oool0oooo00<0
oooo000[0?ooo`030000003oool0oooo00T0oooo0P0000002P2mTJ`0_IFd0;fA[02mUK@0_I6/0;fE
]02mTJ`0_IFd0;fA[02mUK@5000000T0oooo02H0000009aYW02DJI`0W6VL09AYW02LJI`0U6VL0000
002TNI`0[7VT0:AiW02/NJ@0Y7VL0000002LMJ@0Y7FT09aeY02TMJ@0W7FT0:AeY02LMJ@0Y7FT09ae
Y02TMJ@0W7FT0000002/RJ`000000:B5[02/QK@0Y8F/0:b5]02TQJ`0[8Fd0:B5[02/QK@0Y8F/0:b5
]08000000140_Jk50<F^c@000000]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@000000aM[V
0<GJkP35f^H0aM[^0<GJiP03000000030:c^mP2/l_l0[>kf00<0000000T0U8W509b=c@2DRLD0W8g=
0000002DZ^H0U:GV09BZiP2DYNH01P000000302LTLD0W8g50000002LXMD0Y:GN09bQe@2TYMh0W:7E
0:BUgP2LXMD0Y:GN09bQe@<0000000`0Y9G50:BIc@000000Y:GN0:BQe@2TYMh0Y:7E0:BUgP2TXMD0
Y:GN0:BQe@2TYMh30000000;0:bQe@000000[:[N0:BZe@2/Z]h0Y:[E0:bZgP2TZ]D0[:[N0:BZe@2/
Z]h00`000000202/[]h0Y:[E0000002T[]h0[;;N0:B^gP2//]h0Y:kN10000000202/]^H0[;;N0:bf
iP2//]h0[;KV0000002/]^H0Y;KN0`0000002@2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV
0:bjiP03000000080:BniP2/`^h0Y;kV0:c2kP2T_^H0[<;^0:BniP2/`^h3000000080:c6kP2T`^h0
00000:BnkP2/`^h0Y;k^0:c2kP2T_^h4000000<0oooo00<000000?ooo`3oool0303oool00`000000
oooo0?ooo`030?ooo`00;03oool00`000000oooo0?ooo`080?ooo`0;0000002dTJ`0_I6/0;BA[02m
TJ`0]96/0;fA[02dTJ`0_I6/0;BA[02mTJ`01000000;0?ooo`0a0000002DJI@0W6VL09AYU02LJI`0
U6VD0000002TMI`0Y7VT0:AeW02TNJ@0Y7FL0:AiY0000000W7FT09aeW02LMJ@0W7FL09aeY02LMI`0
W7FT09aeW02LMJ@0W7FL0000002/RJ`0]8f/0:b9[0000000Y86/0:b5[02TPJ`0[8F/0:B1[02/QJ`0
Y86/0:b5[0000000_Jk=0;fZa@2m[/d0_J[50000002/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@020000
00040;gFiP35f^h0_MKV0<GJkPP000001`2DRLD50000000:09b=a@2LTLD0W8g509bAa@2LSLD0W975
09b=a@2LTLD0W8g5000000D0W:7E0`000000302LULD0Y9W=09bEa@2TVLd0W9G50000002LXMD0Y:GE
09bQe@2TYMD0W:7E0:BUe@<0000000/0Y:7=0:BMc@2TXLd0Y9g=0:BQc@000000Y:[N0:BUe@2TZ]h0
Y:GE0:BZgP03000000080:bZe@2TZ]D0[:[E0:BZe@2/Z]D000000:bbgP2T[]h3000000090:BbgP2/
]]h0Y;;N0:bfgP2T/]h0[;KN0:BbgP2/]]h0Y;;N00<0000000T0Y;[V0:bjiP2T^^H0[;[V0:BjiP2/
^^H0Y;[V0:bjiP2T^^H00`000000202T`^H0[<K^0000002/_^h0Y;kV0:bnkP2T_^H0[;k^0`000000
202/a^h0Y<;^0:c6kP2T`^h0[<K^0000002/`^h0Y;kV0`0000080?ooo`030000003oool0oooo00/0
oooo00<000000?ooo`3oool00`3oool002d0oooo00<000000?ooo`3oool01`3oool02P000000_IFd
0;fA[02mUK@0_I6/0;fE]02mTJ`0_IFd0;fA[02mUK@3000000d0oooo02l0000009aYW02DJI`0W6VL
09AYW0000000Y7VL0:aiY02TNI`0[7VT0:AiW02/NJ@0Y7VL0000002LMJ@0Y7FT09aeY02TMJ@0W7FT
0:AeY02LMJ@0Y7FT09aeY0000000[8V/0;B=]02/RJ`0]8fd0000002/QK@0Y8F/0:b5]02TQJ`0[8Fd
0:B5[02/QK@000000<F^c@2m[/D0aJk=0;f^a@35[/d0_Jk50000002/VKd0]9W50:bI_@0200000004
0<GJkP35f^H0aM[^0<GJiPH0000000P0W8g=09B9a@2LSLd0U8W509b=c@2DRLD0W8g=09B9a@<00000
0140W86d0000002LSLD0W97509b=a@2LTLD0W8g509bAa@2LSLD0W97509b=a@2LTLD0W8g509bAa@00
0000Y:GN09bQe@030000000<0:BEa@2TVLd0Y9G50:BIc@2TULD0Y9W=0:BEa@2TVLd0Y9G50000002T
XMD0Y:GN0`0000002`2/XMD0Y:7=0:bQe@2TXLd0[:7E0:BQc@2/XMD0Y:7=0000002TZ]D0[:[N00<0
000000P0[:kN0:BZe@2/[]h0Y:[E0:b^gP2TZ]D0[:kN0:BZe@<0000000X0[;;N0:bfiP2//]h0[;KV
0:bbgP2/]^H0[;;N0:bfiP2//]h0[;KV0P000000202/_^H0[<;^0000002/_^H0[;[V0:bniP2/^^H0
[;kV10000000202/`^h0[<K^0:c2kP2/a^h0[<;^0000002T_^H0[<;^0`000000202/a^h0Y<;^0:c6
kP2T`^h0[<K^0:C2kP2/a^h0Y<;^0`00000<0?ooo`030000003oool0oooo00X0oooo00<000000?oo
o`3oool00`3oool002h0oooo0P0000080?ooo`080000002mTJ`0]96/0;fA[02dTJ`0_I6/0;BA[02m
TJ`2000000l0oooo0280000009AYU02LJI`0U6VD0000002TMI`0Y7VT0:AeW02TNJ@0Y7FL0:AiY02T
MI`0Y7VT0000002LMJ@0W7FL09aeY02LMI`0W7FT09aeW02LMJ@0W7FL0000002/RJ`0]8f/0:b9[02d
SJ`0[8V/0000002TPJ`0[8F/0:B1[02/QJ`0Y86/0P0000002@2m[/d0_J[50;f^c@2mZ/D0_Jk=0;fZ
a@2m[/d000000;BIa@02000000030;gFiP35f^h0_MKV00<0000000H0U7V/09ai]02DNJ`0W7Vd09Ai
[000000509B9a@<000001@2LPK@030000000W8g509bAa@2LSLD0W97509b=a@2LTLD0W8g509bAa@2L
SLD0W97509b=a@<0000000`0W9G50:BIc@2LULD0Y9W=09bEa@2TVLd0W9G50:BIc@2LULD0Y9W=09bE
a@2TVLd30000000;0:BQc@2TWLd0Y:7=0:BMc@2TXLd0Y9g=0:BQc@2TWLd0Y:7=0:BMc@2TXLd00`00
00002@2/Z]D0Y:[E0:bZe@2TZ]D0[:[E0:BZe@2/Z]D0Y:[E0:bZe@0200000080[;KN00T000000:bf
gP2T/]h0[;KN0:BbgP2/]]h0Y;;N0:bfgP2T/]h00P0000040:bniP030000002T^^H0[;[V00<00000
00T0[<K^0:C2iP2/a^h0Y<;V0:c6kP2T`^H0[<K^0:C2iP2/a^h00`0000001`2/a^h0Y<;^0:c6kP2T
`^h0[<K^0:C2kP2/a^h01000000@0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0
0`3oool00300oooo00<000000?ooo`3oool01@3oool020000000_I6/0;fE]02mTJ`0_IFd0;fA[02m
UK@000004@3oool050000000W6VL09AYW0000000Y7VL0:aiY02TNI`0[7VT0:AiW02/NJ@0Y7VL0:ai
Y02TNI`0000009aeY02TMJ@0W7FT0:AeY02LMJ@0Y7FT0P0000006P2/RJ`0]8fd0:b9[02dSK@0[8V/
0;B=]0000000[8Fd0:B5[02/QK@0Y8F/0000002m[/D0aJk=0;f^a@35[/d0_Jk50<F^c@2m[/D0aJk=
0;f^a@35[/d000000<GJkP35f^H0aM[^0P0000002`2DNJ`0W7Vd09Ai[02LNK@0U7V/09ai]02DNJ`0
W7Vd09Ai[0000000U8W500<000000140W86d0:B5_@2LPK@0Y8Fm09b1]02TQKd0W86d0:B5_@000000
W97509b=a@2LTLD0W8g509bAa@2LSLD0W97509b=a@030000000=0:B=_@2TSLD000000:BIc@2TULD0
Y9W=0:BEa@2TVLd0Y9G50:BIc@2TULD0Y9W=0:BEa@020000000<0:bMc@000000[:7E0:BQc@2/XMD0
Y:7=0:bQe@2TXLd0[:7E0:BQc@2/XMD0Y:7=0P0000002P2/Z]D000000:b^gP2TZ]D0[:kN0:BZe@2/
[]h0Y:[E0:b^gP2TZ]D20000000:0:bfiP2/]]h0[;KV0:bfgP000000[;;N0:bfiP2//]h0[;KV0:bb
gP<0000000H0[;kV0:c2kP2/_^H0[<;^0:bniP2/`^h3000000090:c6kP2/`^h0[<K^0:c2kP2/a^h0
[<;^0:c6kP2/`^h0[<K^00<0000000H0[<kf0:C:mP000000Y<;^0:c6kP2T`^h4000001@0oooo00<0
00000?ooo`3oool02@3oool00`000000oooo0?ooo`030?ooo`00<@3oool00`000000oooo0?ooo`04
0?ooo`070000002mTJ`0]96/0;fA[02dTJ`0_I6/0000000B0?ooo`0Z0000002DJI@0W6VL0000002T
NJ@0Y7FL0:AiY02TMI`0Y7VT0:AeW02TNJ@0Y7FL0:AiY0000000W7FT09aeW02LMJ@0W7FL09aeY000
0000]8f/0:b9[02dSJ`0[8V/0;B=[02/RJ`0]8f/0:b9[02dSJ`000000:b5[02TPJ`000000;fZa@2m
[/d0_J[50;f^c@2mZ/D0_Jk=0;fZa@2m[/d0_J[51P0000002P2LNK@0U7V/09ai]02DNJ`0W7Vd09Ai
[02LNK@0U7V/09ai]02DNJ`3000000/0W86d00H0000009b=a@2LTLD0W8g509bAa@2LSLD3000000D0
Y8fm00T0000009bEa@2TVLd0W9G50:BIc@2LULD0Y9W=09bEa@2TVLd00P000000302TVLD0[9W=0:BI
a@2/VLd000000:BQc@2TWLd0Y:7=0:BMc@2TXLd0Y9g=0:BQc@80000000T0[:G=0:bZe@2/YLd0[:[E
0000002/Z]D0Y:[E0:bZe@2TZ]D00`0000070:bfgP030000002T/]h0[;KN008000001P2/_^H30000
00080:c>mP2Tb^h000000:C2iP2/a^h0Y<;V0:c6kP2T`^H4000000040:C:mP2Tb^h0Y<[f0:C:kPD0
00006@3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo00<0oooo000b0?ooo`030000
003oool0oooo00@0oooo00D000000;fE]02mTJ`0_IFd0000000C0?ooo`0M0000002LJI`000000:ai
Y02TNI`0[7VT0:AiW02/NJ@0Y7VL0:aiY02TNI`0[7VT0:AiW0000000W7FT0:AeY02LMJ@0Y7FT0000
002dSK@0[8V/0;B=]02/RJ`0]8fd0:b9[02dSK@0[8V/0;B=]02/RJ`00`0000002@2m[/D0aJk=0;f^
a@35[/d0_Jk50<F^c@2m[/D0aJk=0;f^a@0400000080W76T00T0000009Ai[02LNK@0U7V/09ai]02D
NJ`0W7Vd09Ai[02LNK@00P0000004@2LOJ`0000009b1]02TQKd0W86d0:B5_@2LPK@0Y8Fm09b1]02T
QKd0W86d0:B5_@2LPK@0Y8Fm09b1]0000000W8g500<0000000h0Y8fm0:B=a@2TSKd0Y8g50:B=_@2T
SLD0Y8fm0:B=a@2TSKd000000:BEa@2TVLd0Y9G50:BIc@<0000000/0[9g=0:BIa@2/WLd0Y9W50:bM
c@2TVLD000000:BQc@2/XMD0Y:7=0:bQe@03000000H0[:[E00<000000:BZe@2/[]h00P0000002P2/
]]h0[;KV0:bfgP2/]^H0[;KN0:bfiP2/]]h0[;KV0:bfgP2/]^H3000000050:c2kP2/_^H0[<;^0:bn
iP2/`^h00`000000202/c_H0Y<k^0:c>mP2Tc^h0[<kf0000002/a^h0[<;^0`000000102Tb_H0[<kf
0:C:mP2/c_H4000001l0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`020?ooo`00
<`3oool2000000@0oooo00D000000;BA[02mTJ`0]96/0000000C0?ooo`80000002D0Y7VT0:AeW02T
NJ@0Y7FL0:AiY02TMI`0Y7VT0:AeW02TNJ@0Y7FL0:AiY02TMI`0000009aeW02LMJ@000000;B=[02/
RJ`0]8f/0:b9[02dSJ`0[8V/0;B=[02/RJ`0]8f/0:b9[02dSJ`000000;f^c@2mZ/D0_Jk=0;fZa@2m
[/d0_J[50;f^c@2mZ/D0_Jk=00<0000000`0W76T09AaY02LLJ@0U76T09aaY0000000W7Vd09Ai[02L
NK@0U7V/09ai]02DNJ`2000000050:Am[02LNJ@0Y7f/09aiY0000000302LPK@2000000`0Y8fm00<0
00000:BIc@2LULD00P000000302/VLd0Y9W50:bIc@2TVLD0[9W=0:BIa@2/VLd0Y9W50:bIc@2TVLD0
00000:BMc@80000000T0[:[E0:bUc@2/Z]D0[:G=0:bZe@2/YLd0[:[E0:bUc@2/Z]D00`0000090:bf
gP<0000000<0[<[^0:c6kP0000000P2/_^H3000000080:c>mP2Tb^h0[<kf0:C:kP2/c_H0Y<[^0:c>
mP2Tb^h3000000030:C:kP2Tb_H0Y<[^00@00000903oool00`000000oooo0?ooo`070?ooo`030000
003oool0oooo0080oooo000e0?ooo`080000003oool0oooo0?ooo`000000_IFd0;fA[000000D0?oo
o`0W0000002/NJ@0Y7VL0:aiY02TNI`0[7VT0:AiW02/NJ@0Y7VL0:aiY02TNI`0[7VT0:AiW02/NJ@0
00000:AeY0000000]8fd0:b9[02dSK@0[8V/0;B=]02/RJ`0]8fd0:b9[02dSK@0[8V/0;B=]0000000
g_[N00000035[/d0_Jk50<F^c@2m[/D0aJk=0;f^a@35[/d0000000T0W76T00@0000009ai]02DNJ`0
W7Vd0P0000004@2LOJ`0Y7f/09am[02TOJ`0W7f/0:Am[0000000Y8Fm09b1]02TQKd0W86d0:B5_@2L
PK@0Y8Fm09b1]02TQKd0W86d00<0000000h0Y8fm0:B=a@2TSKd0Y8g50:B=_@2TSLD0Y8fm0:B=a@2T
SKd0Y8g50:B=_@2TSLD0Y8fm0:B=a@80000000/0Y9W50:bMc@2TVLD0[9g=0:BIa@2/WLd0Y9W50:bM
c@2TVLD0[9g=0:BIa@03000000P0[:[E0`000000202d`^H0[;kV0000002/]]h0[;KV0:bfgP2/]^H0
[;KN100000050:c:kP<0000000L0[<kf0:C>kP2/c_H0Y<k^0:c>mP2Tc^h0[<kf00@0000000<0Y=;f
0:CFo`0000000`00000X0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool00P3oool0
03H0oooo00@000000?ooo`3oool0oooo0P00000E0?ooo`0>0000002TMI`0Y7VT0:AeW02TNJ@0Y7FL
0:AiY02TMI`0Y7VT0:AeW02TNJ@0Y7FL0:AiY02TMI`20000000D0;B=[02/RJ`0]8f/0:b9[02dSJ`0
[8V/0;B=[02/RJ`0]8f/0:b9[02dSJ`000000=kjgP3Em]h000000;fZa@2m[/d0_J[50;f^c@2mZ/D2
0000000<09aaY02DLJ@0W76T09AaY02LLJ@0U76T09aaY02DLJ@0W76T09AaY0000000U7V/0P000000
2@2TOJ`0W7VT0:Am[02LNJ@0Y7f/09aiY02TOJ`0W7VT0000000809b1]080000000@0Y8Fd0:B9]02T
QK@000002`2TSKd30000000:0:BIa@2/VLd0Y9W50:bIc@2TVLD0[9W=0:BIa@2/VLd0Y9W50:bIc@80
000000P0[:kE0;B^e@000000[:[E0:bUc@2/Z]D0[:G=0:bZe@<0000000H0];kV0:bngP2d_^H0[;kN
0;BniP0000020:bfgP<0000000H0[<K^0:c:kP2/a^h0[<[^0:c6kP2/b^h3000000060:CJo`2Lf_H0
00000:C:kP2/c_H0Y<[^100000000`2Te_l0W=;f0:CFo`03000002d0oooo00<000000?ooo`3oool0
1P3oool00`000000oooo0?ooo`020?ooo`00=`3oool010000000oooo0?ooo`00000F0?ooo`0T0000
002/NJ@0Y7VL0:aiY02TNI`0[7VT0:AiW02/NJ@0Y7VL0:aiY02TNI`0[7VT0:AiW02/NJ@000000;B=
]02/RJ`0]8fd0:b9[02dSK@0[8V/0;B=]02/RJ`0]8fd0:b9[02dSK@000000=kjgP3Em]h0g_[N0=Gf
gP000000_Jk50<F^c@2m[/D00000302LLJ@20000000A09am[02TOJ`0W7f/0:Am[02LOJ`0Y7f/09am
[02TOJ`0W7f/0:Am[02LOJ`0000009b1]02TQKd0W86d0:B5_@2LPK@00P0000003`2/RKd0Y8Vd0:b9
_@2TRK@0[8Vm0000002TSLD0Y8fm0:B=a@2TSKd0Y8g50:B=_@2TSLD0Y8fm0:B=a@02000000<0[9W5
00P000000:BIa@2/WLd0Y9W50:bMc@2TVLD0[9g=0:BIa@80000000D0]:kE0:b^e@2d[]D0[:kE0000
00020:bZe@<0000000P0]<;V0:bniP2d`^H0[;kV0;C2iP2/_^H0]<;V0:bniP<000001P2/b^h30000
00050:CNo`2Tf_l0Y=ko0:CJo`2Tg_l010000000102Td_H0Y=Ko0:CBmP2Te_l300000340oooo00<0
00000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`00>03oool2000001L0oooo00d00000
0:AeW02TNJ@0Y7FL0:AiY02TMI`0Y7VT0:AeW02TNJ@0Y7FL0:AiY02TMI`0Y7VT008000000180[8V/
0;B=[02/RJ`0]8f/0:b9[02dSJ`0[8V/0;B=[02/RJ`0]8f/0000003Nn]h0eOKN0=kjgP3Em]h0g_[N
0000002m[/d30000000;09aaY02DLJ@0W76T09AaY02LLJ@0U76T09aaY02DLJ@0W76T09AaY02LLJ@0
0P0000003@2LNJ@0Y7f/09aiY02TOJ`0W7VT0:Am[02LNJ@0Y7f/09aiY02TOJ`0W7VT0:Am[0000000
0`2LPK@2000000090:B5]02TRK@0Y8Fd0:B9]02TQK@0Y8Vd0:B5]02TRK@0000000H0Y8fm0P000005
0:bIa@050000002/VLd0Y9W50:bIc@2TVLD00`0000001P2/[]D0]:kE0:b^e@2d[]D0[:kE0;B^e@<0
000000P0];kV0:bngP2d_^H0[;kN0;BniP2/_]h0];kV0:bngP<0000000H0Y=[f0:cNo`0000000000
0:c6kP2/b^h3000000040:CJo`2Lf_H0Y=[o09cJmP@0000000D0U=ko09CJo`000000W=;f0:CFo`03
000003@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`020?ooo`00>P3oool00`00
0000oooo0?ooo`0D0?ooo`0T0000002/NJ@0Y7VL0:aiY02TNI`0[7VT0:AiW02/NJ@0Y7VL0:aiY02T
NI`0[7VT0000003=Z[d0aJZd0000002/RJ`0]8fd0:b9[02dSK@0[8V/0;B=]02/RJ`0]8fd0000003N
n]h0eOKN0=kjgP3Em]h0g_[N0=GfgP3Nn]h0000009aaY02LKI`000002@2LLJ@20000000?0:Am[000
0000Y7f/09am[02TOJ`0W7f/0:Am[02LOJ`0Y7f/09am[02TOJ`0W7f/0:Am[02LOJ`0Y7f/00<00000
00l0[8Vm0:B9]02/RKd0Y8Vd0:b9_@2TRK@0[8Vm0:B9]02/RKd0Y8Vd0000002TSKd0Y8g50:B=_@2T
SLD00P0000080:bIa@0;0000002/WLd000000000002/[]D0]:kE0:b^e@2d[]D0[:kE0;B^e@2/[]D0
0P000000202de_H0]=K^0000002/_^H0]<;V0:bniP2d`^H0[;kV0`0000001@2Tg_H0[=ko0:CNmP2/
g_l0Y=kf00@0000000<0Y=ko0:CJo`2Tg_l01000000509CNo`@00000>03oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo0080oooo000k0?ooo`030000003oool0oooo01@0oooo01d00000
0:AiY02TMI`0Y7VT0:AeW02TNJ@0Y7FL0:AiY02TMI`0Y7VT0:AeW0000000aJFd0<fZ_@000000]8f/
0:b9[02dSJ`0[8V/0;B=[02/RJ`0]8f/0000003Nn]h0eOKN0=kjgP3Em]h0g_[N0=GfgP020000000^
09aaW02LKI`0W76L0000002LLJ@0U76T09aaY02DLJ@0W76T09AaY02LLJ@0U76T0000002TNJ@0Y7fT
0:AiY02TOJ@000000:Am[02LNJ@0Y7f/09aiY02TOJ`0W7VT0:Am[02LNJ@0Y7f/09aiY02TOJ`0W7VT
0000002TQK@0Y8Vd0:B5]02TRK@0Y8Fd0:B9]02TQK@0Y8Vd0:B5]02TRK@0Y8Fd0:B9]02TQK@00000
0:B=_@8000002P2/VLD2000000070;B^e@2/[]D0]:kE0:b^e@2d[]D0[:kE0;B^e@02000000070:cF
kP2de_H0[=K^0;CFmP000000];kV0:bngP03000000040:CJmP2/g_l0Y=[f0:cNo`@0000000<0Nn;f
08?RmP1kh_H0100000000`2Dg_l0U=[o09CNo`06000003d0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`010?ooo`00?03oool00`000000oooo0?ooo`0C0?ooo`0K0000002TNI`0[7VT
0:AiW02/NJ@0Y7VL0:aiY02TNI`0[7VT0:AiW0000000aJZd0<fZ_@35Z[@000000:b9[02dSK@0[8V/
0;B=]02/RJ`0]8fd0000003Nn]h0eOKN0=kjgP3Em]h0g_[N0080000000H0W6fL09aaY02LKI`0W76T
09a]W000000709aaY00A0000002TNJ@0Y7f/0:AiY02TOJ`0Y7VT0000002LOJ`0Y7f/09am[02TOJ`0
W7f/0:Am[02LOJ`0Y7f/09am[02TOJ`00`0000003@2/RKd0Y8Vd0:b9_@2TRK@0[8Vm0:B9]02/RKd0
Y8Vd0:b9_@2TRK@0[8Vm0:B9]02/RKd00P00000:0:bIa@80000000H0];kN0000002/[]D0]:kE0:b^
e@2d[]D3000000060;CFmP2de^h0]=Kf0;CFkP2de_H0]=K^0`0000000`2Tg_H0[=ko0:CNmP040000
000308?Ro`23h_H0Pn;o00@0000000<0Lm[f000000000000100000140?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool00@3oool003d0oooo0P00000C0?ooo`0X0000002TNJ@0Y7FL0:Ai
Y02TMI`0Y7VT0:AeW02TNJ@0Y7FL00000035YK@0cJZm0<FU]03=Z[d0aJFd0000002/RJ`0]8f/0:b9
[02dSJ`000000=kjgP3Em]h0g_[N0=GfgP3Nn]h0000009aaW02LKI`0W76L09a]W02LLI`0W6fL09aa
W0000000W76T09AaY02LLJ@0U76T09aaY080000001l0Y7VT0:AmY02TNJ@0Y7fT0:AiY02TOJ@00000
0:Am[02LNJ@0Y7f/09aiY02TOJ`0W7VT0:Am[02LNJ@0Y7f/0000002/RK@0[8V/0000002TQK@0Y8Vd
0:B5]02TRK@0Y8Fd0:B9]02TQK@0Y8Vd0:B5]02TRK@0Y8Fd00<000002@2/VLD2000000060;g2gP2d
_]h0_L;N0000002d[]D0[:kE0P0000001P2de_H0[=K^0;CFmP2/e^h0]=Kf0:cFkP<0000000<0J]kN
07?NiP0000000`0000000`1kh_H0Pn;f07_RmP09000004X0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`010?ooo`00?`3oool00`000000oooo0?ooo`0@0?ooo`0H0000002TNI`0[7VT
0:AiW02/NJ@0Y7VL0:aiY02TNI`000000<FZ]03=Z[d0aJZd0<fZ_@35Z[@0cJZm0000002dSK@0[8V/
0;B=]0000000g_[N0=GfgP3Nn]h0eOKN0P0000002P2LLJ@0W6fL09aaY02LKI`0W76T09a]W02LLJ@0
W6fL09aaY000000309aaY00A0000002TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0Y7VT0000002L
OJ`0Y7f/09am[02TOJ`0W7f/0:Am[02LOJ`00P0000003@2/SK@0[8Vd0:b=]02/RK@000000:B9]02/
RKd0Y8Vd0:b9_@2TRK@0[8Vm0:B9]02/RKd00P000000102dXLD0]:G=0;BQa@0000060:bIa@800000
00D0];kN0;g2iP2d_]h0_L;V0;BngP03000000050;CFkP2de_H0]=K^0;CFmP2de^h00`0000000`1Z
g]h000000000000<00000540oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`010?oo
o`00@03oool00`000000oooo0?ooo`0?0?ooo`0F0000002TNJ@0Y7FL0:AiY02TMI`0Y7VT0:AeW02T
NJ@000000<fZ_@35YK@0cJZm0<FU]03=Z[d0aJFd0000002/RJ`0]8f/0000003Nn]h0eOKN0=kjgP80
000002l0W6fL09aaW02LKI`0W76L09a]W02LLI`0W6fL09aaW02LKI`0W76L09a]W0000000U76T09aa
Y0000000Y7fT0:AiY02TOJ@0Y7VT0:AmY02TNJ@0Y7fT0:AiY02TOJ@000000:Am[02LNJ@0Y7f/09ai
Y02TOJ`0W7VT0000002/RJ`0[8Vd0:b9[02/RK@0[8V/0:b9]0000000Y8Vd0:B5]02TRK@0Y8Fd0:B9
]02TQK@0Y8Vd000000050;BQa@030000002/VLD0[9W50080[9W50P0000001@2m`]h0];kN0;g2gP2d
_]h0_L;N008000000P1ki]h00`000000]=Kf0:cFkP0?000005P0oooo00D000000?ooo`3oool0oooo
000000030?ooo`00@@3oool00`000000oooo0?ooo`0>0?ooo`0@0000002TNI`0[7VT0:AiW02/NJ@0
Y7VL0:aiY0000000cJZm0<FZ]03=Z[d0aJZd0<fZ_@35Z[@0cJZm0<FZ]080000000<0g_[N0=GfgP00
00000P000000302LKI`0W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T09a]W02LLJ@0W6fL09aaY0<0
000000l0Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0Y7VT0000002LOJ`0Y7f/09am
[02TOJ`00P0000003@2/RK@0[8fd0:b9]02/SK@0[8Vd0:b=]02/RK@000000:B9]02/RKd0Y8Vd0:b9
_@2TRK@00P0000002@2dYLd0]:750;BUc@2dXLD0]:G=0;BQa@2dYLd000000:bIa@02000000050;Bn
gP2m`^H0];kN0;g2iP2d_]h00P0000001023j^H0NnKN08?ZiP1ki]h<000005l0oooo00@000000?oo
o`3oool000000`3oool00480oooo0P00000>0?ooo`0@0000002TNJ@0Y7FL0:AiY02TMI`0Y7VT0000
003=Z[d0aJFd0<fZ_@35YK@0cJZm0<FU]03=Z[d0aJFd0<fZ_@@000000P3oool0;0000000W76L09a]
W02LLI`0W6fL09aaW02LKI`0W76L09a]W02LLI`0W6fL09aaW02LKI`000000:AiY02TOJ@0Y7VT0:Am
Y02TNJ@0Y7fT0:AiY02TOJ@0Y7VT0:AmY02TNJ@0Y7fT0:AiY0000000W7VT0:Am[0000000[8Vd0:b9
[02/RK@0[8V/0:b9]02/RJ`0[8Vd0:b9[02/RK@000000:B9]02TQK@0Y8Vd0P0000090;BQa@800000
00@0_L;N0;BngP2m`]h0];kN0`00000207_VgPL00000J03oool010000000oooo0?ooo`0000030?oo
o`00A03oool00`000000oooo0?ooo`0;0?ooo`0?0000002TNI`0[7VT0:AiW02/NJ@000000<fZ_@35
Z[@0cJZm0<FZ]03=Z[d0aJZd0<fZ_@35Z[@0cJZm00<00000103oool030000000W6fL09aaY02LKI`0
W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T09a]W080000000d0Y7f/0:AiY02TOJ`0Y7VT0:Am[02T
NJ@0Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@0Y7f/00<0000000/0[8fd0:b9]02/SK@0[8Vd0:b=]02/
RK@0[8fd0:b9]02/SK@0[8Vd0:b=]003000000090;BUc@2dXLD0]:G=0;BQa@2dYLd0]:750;BUc@2d
XLD0]:G=0080000000@0]?;^0000002d_]h0_L;V0P0000000`1ki]h000000000000200000700oooo
00<000000?ooo`0000000`3oool004D0oooo00<000000?ooo`3oool02P3oool03`000000Y7VT0:Ae
W02TNJ@0Y7FL00000035YK@0cJZm0<FU]03=Z[d0aJFd0<fZ_@35YK@0cJZm000000070?ooo`0Z0000
002LLI`0W6fL09aaW02LKI`0W76L09a]W02LLI`0W6fL09aaW02LKI`000000:amY0000000Y7VT0:Am
Y02TNJ@0Y7fT0:AiY02TOJ@0Y7VT0:AmY02TNJ@0Y7fT0:AiY02TOJ@0Y7VT0000002/RJ`0[8Vd0:b9
[02/RK@0[8V/0:b9]02/RJ`0[8Vd0:b9[02/RK@0[8V/0:b9]02/RJ`000002@2dXLD2000000030;gf
mP2dl^h0_OKf00H00000M@3oool00`000000oooo000000020?ooo`00AP3oool00`000000oooo0?oo
o`0:0?ooo`0=0000002/NJ@0Y7VL00000035Z[@0cJZm0<FZ]03=Z[d0aJZd0<fZ_@35Z[@0cJZm0000
00080?ooo`0J0000002LKI`0W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T09a]W02LLJ@000000:b1
Y0000000Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@20000000<
0:b=]02/RK@0[8fd0:b9]02/SK@0[8Vd0:b=]02/RK@0[8fd0:b9]02/SK@0[8Vd0P0000001`2dYLd0
]:750;BUc@2dXLD0]:G=0;BQa@2dYLd00P0000000`2dl^h0_OKf0;CbkP04000007X0oooo0P000002
0?ooo`00A`3oool00`000000oooo0?ooo`090?ooo`0<0000002TMI`000000<FU]03=Z[d0aJFd0<fZ
_@35YK@0cJZm0<FU]03=Z[d000002@3oool09P000000W76L09a]W02LLI`0W6fL09aaW02LKI`0W76L
09a]W02LLI`000000:b1Y02/OJ@0[86T0000002TOJ@0Y7VT0:AmY02TNJ@0Y7fT0:AiY02TOJ@0Y7VT
0:AmY02TNJ@000000;BE]0000000[8V/0:b9]02/RJ`0[8Vd0:b9[02/RK@0[8V/0:b9]02/RJ`0[8Vd
0P0000000`35`]D0cLKE000000040;BQa@80000000<0_OKf0;CbkP2mm_H00P00001n0?ooo`800000
0P3oool004P0oooo0P0000090?ooo`80000000T0aJZd0<fZ_@35Z[@0cJZm0<FZ]03=Z[d0aJZd0<fZ
_@0000002`3oool0;0000000W76T09a]W02LLJ@0W6fL09aaY02LKI`0W76T0000002/PJ@0[7fT0:b1
Y02/OJ@000000:AiY02TOJ`0Y7VT0:Am[02TNJ@0Y7f/0:AiY02TOJ`0Y7VT0000002dUK@0_IVm0;BE
]0000000[8Vd0:b=]02/RK@0[8fd0:b9]02/SK@0[8Vd0:b=]0000000cLKN0<G6e@3=a]h0aLKE0000
002dXLD0]:G=0P0000000`2dl^h0_OKf0;CbkP0200000800oooo0P0000020?ooo`00BP3oool00`00
0000oooo0?ooo`060?ooo`80000000H0cJZm0<FU]03=Z[d0aJFd0<fZ_@35YK@2000000`0oooo02X0
000009a]W02LLI`0W6fL09aaW02LKI`0W76L0000002/PJ@0[7fT0:b1Y02/OJ@0[86T0000002TOJ@0
Y7VT0:AmY02TNJ@0Y7fT0:AiY02TOJ@0Y7VT0000002dUK@0_IVd0;BE]02mVK@000000:b9]02/RJ`0
[8Vd0:b9[02/RK@0[8V/0:b9]0000000cLKE0<G2e@3=a]D0aL;E0<g6e@35`]D2000000030;gfmP2d
l^h0_OKf00800000OP3oool4000000@0oooo001;0?ooo`030000003oool0oooo00D0oooo00P00000
0<fZ_@35Z[@0cJZm0<FZ]03=Z[d0aJZd000000h0oooo0240000009aaY02LKI`0W76T09a]W02LLJ@0
00000:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0000002TNJ@0Y7f/0:AiY02TOJ`0Y7VT0:Am[02TNJ@0
00000;BE]02mVKd0]9Fd0;fI_@2dUK@000000:b9]02/SK@0[8Vd0:b=]02/RK@00P0000001@3=a]h0
aLKE0<g6gP35a]D0cLKN0080000000<0:@000000000000000P00001l0?ooo`@00000203oool004`0
oooo00<000000?ooo`3oool0103oool01`000000aJFd0<fZ_@35YK@0cJZm0<FU]00000003`3oool0
9`000000W6fL09aaW02LKI`0W76L09a]W0000000[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@000000:Am
Y02TNJ@0Y7fT0:AiY02TOJ@0Y7VT0000002dUK@0_IVd0;BE]02mVK@0]9Fd0;fI]0000000[8Vd0:b9
[02/RK@0[8V/00000035`]D0cLKE0<G2e@3=a]D0aL;E0<g6e@04000007`0oooo1000000<0?ooo`00
C@3oool2000000D0oooo00D000000<FZ]03=Z[d0aJZd0000000@0?ooo`0U0000002LLJ@0W6fL09aa
Y02LKI`000000:amY02/PJ@0[7fT0:b1Y02/OJ@0[86T0:amY0000000Y7VT0:Am[02TNJ@0Y7f/0:Ai
Y0000000]9Fd0;fI_@2dUK@0_IVm0;BE]02mVKd0]9Fd0;fI_@000000[8fd0:b9]0000000aLKE0<g6
gP35a]D0cLKN0<G6e@02000007`0oooo1000000@0?ooo`00C`3oool00`000000oooo0?ooo`020?oo
o`040000003=Z[d0aJFd00000140oooo01`0000009a]W02LLI`0W6fL0000002/OJ@0[86T0:amY02/
PJ@0[7fT0:b1Y02/OJ@0[86T0000002TOJ@0Y7VT0:AmY02TNJ@000000;BE]02mVK@0]9Fd0;fI]02d
UK@0_IVd0;BE]02mVK@0]9Fd0`0000001035`]D0cLKE0<G2e@3=a]D2000007/0oooo0`00000D0?oo
o`00D03oool00`000000oooo0?ooo`020?ooo`8000004P3oool08`000000W76T09a]W0000000[7fT
0:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0:b1Y02/OJ@0[86T0000002TOJ`0Y7VT0000002dUK@0_IVm
0;BE]02mVKd0]9Fd0;fI_@2dUK@0_IVm0;BE]02mVKd000000<g6gP35a]D0cLKN0<G6e@3=a]h00000
07T0oooo1000000G0?ooo`00D@3oool01@000000oooo0?ooo`3oool0000001<0oooo01/0000009a]
W0000000[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0:b1Y02/OJ@0[86T0:amY0000000Y7VT0000
002dUK@0_IVd0;BE]02mVK@0]9Fd0;fI]02dUK@0_IVd0;BE]02mVK@0100000000`35`]D000000000
001f0?ooo`@000006`3oool00580oooo0P00000E0?ooo`0>0000002LLJ@000000:b1Y02/OJ@0[86T
0:amY02/PJ@0[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@20000000:0;BE]02mVKd0]9Fd0;fI_@2dUK@0
_IVm0;BE]02mVKd0]9Fd0;fI_@H00000M03oool4000001l0oooo001D0?ooo`030000003oool0oooo
0180oooo0P0000005`2/PJ@0[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0:b1Y02/OJ@0[86T0:am
Y0000000]9Fd0;fI]02dUK@0_IVd0;BE]02mVK@0]9Fd0;fI]02dUK@0_IVd00D00000LP3oool40000
02<0oooo001E0?ooo`030000003oool0oooo0140oooo00d000000:b1Y02/OJ@0[86T0:amY02/PJ@0
[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0080000000T0_IVm0;BE]02mVKd0]9Fd0;fI_@2dUK@0
_IVm0;BE]02mVKd01000001a0?ooo`<000009`3oool005H0oooo00<000000?ooo`3oool0403oool0
5P000000[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0:b1Y02/OJ@0[86T0:amY0000000g];50=kB
c@000000_IVd0;BE]02mVK@0]9Fd0;fI]02dUK@4000006l0oooo1000000Z0?ooo`00E`3oool20000
0100oooo01D000000:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@00000
0>KBc@3Nd/d000000;BE]02mVKd0]9Fd0;fI_@2dUK@00P00001^0?ooo`@00000;P3oool005T0oooo
00<000000?ooo`3oool03@3oool050000000[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0:b1Y02/
OJ@0[86T0000003Nd/d0g];50=kBc@3Nd/D000000;BE]02mVK@0]9Fd0P00001[0?ooo`@00000<P3o
ool005X0oooo00<000000?ooo`3oool03@3oool05@000000[7fT0:b1Y02/OJ@0[86T0:amY02/PJ@0
[7fT0:b1Y0000000i];=0=kBc@3Vd/d0g];=0>KBc@000000_IVm0;BE]0000000oooo0000001W0?oo
o`@00000=P3oool005/0oooo00<000000?ooo`3oool0303oool040000000[86T0:amY02/PJ@0[7fT
0:b1Y02/OJ@0[86T0000003Nd/d0g];50=kBc@3Nd/D0g];=0=kBa@3Nd/d200000080oooo00<00000
0?ooo`3oool0H@3oool4000003X0oooo001L0?ooo`800000303oool04@000000[7fT0:b1Y02/OJ@0
[86T0:amY02/PJ@0[7fT0000003Nd/d0i];=0=kBc@3Vd/d0g];=0>KBc@3Nd/d0000000<0oooo00<0
00000?ooo`3oool0GP3oool3000003h0oooo001N0?ooo`030000003oool0oooo00T0oooo01000000
0:b1Y02/OJ@0[86T0:amY02/PJ@0[7fT0000003Nd/D0g];=0=kBa@3Nd/d0g];50=kBc@3Nd/D00000
103oool00`000000oooo0?ooo`1J0?ooo`@00000@@3oool005l0oooo00<000000?ooo`3oool0203o
ool03`000000[7fT0:b1Y02/OJ@0[86T0:amY0000000g];=0>KBc@3Nd/d0i];=0=kBc@3Vd/d0g];=
000000050?ooo`030000003oool0oooo05H0oooo100000150?ooo`00H03oool00`000000oooo0?oo
o`070?ooo`0<0000002/PJ@0[7fT0:b1Y02/OJ@000000=kBa@3Nd/d0g];50=kBc@3Nd/D0g];=0P00
00060?ooo`030000003oool0oooo0580oooo100000190?ooo`00H@3oool2000000L0oooo00`00000
0:amY02/PJ@0[7fT0:b1Y0000000i];=0=kBc@3Vd/d0g];=0>KBc@0000080?ooo`030000003oool0
oooo04h0oooo1000001=0?ooo`00H`3oool00`000000oooo0?ooo`040?ooo`0;0000002/PJ@0[7fT
0:b1Y0000000g];=0=kBa@3Nd/d0g];50=kBc@0000002@3oool00`000000oooo0?ooo`1;0?ooo`<0
0000D@3oool006@0oooo00<000000?ooo`3oool0103oool02@000000[86T0000003Vd/d0g];=0>KB
c@3Nd/d0i];=0000000:0?ooo`030000003oool0oooo04L0oooo1000001D0?ooo`00I@3oool00`00
0000oooo0?ooo`030?ooo`80000000H0g];=0=kBa@3Nd/d0g];50=kBc@00000;0?ooo`030000003o
ool0oooo04<0oooo1000001H0?ooo`00IP3oool2000000<0oooo0P0000000`3Nd/d0i];=0=kBc@02
000000`0oooo00<000000?ooo`3oool0?`3oool4000005`0oooo001X0?ooo`030000003oool0oooo
0080000000<0g];=0=kBa@0000003P3oool00`000000oooo0?ooo`0k0?ooo`@00000H03oool006T0
oooo00@000000?ooo`3oool0oooo0P00000?0?ooo`030000003oool0oooo03P0oooo0`00001T0?oo
o`00JP3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo03@0oooo1000001W0?ooo`00
J`3oool200000100oooo00<000000?ooo`3oool0<@3oool4000006/0oooo001]0?ooo`030000003o
ool0oooo00d0oooo00<000000?ooo`3oool0;@3oool4000006l0oooo001^0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0:@3oool4000007<0oooo001_0?ooo`030000003oool0oooo
00/0oooo00<000000?ooo`3oool09P3oool3000007L0oooo001`0?ooo`8000002`3oool00`000000
oooo0?ooo`0R0?ooo`@00000NP3oool00780oooo00<000000?ooo`3oool0203oool00`000000oooo
0?ooo`0N0?ooo`@00000OP3oool007<0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?oo
o`0J0?ooo`@00000PP3oool007@0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0F
0?ooo`@00000QP3oool007D0oooo0P0000060?ooo`030000003oool0oooo0180oooo1000002:0?oo
o`00M`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo00l0oooo0`00002>0?ooo`00
N03oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00/0oooo1000002A0?ooo`00N@3o
ool01@000000oooo0?ooo`3oool0000000T0oooo1000002E0?ooo`00NP3oool2000000030?ooo`00
0000oooo00@0oooo1000002I0?ooo`00O03oool2000000030?ooo`000000000000800000W@3oool0
07d0oooo0P00002Q0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {232.562, 0}} -> {-0.0710017, 0.0132871, \
0.00358137, 0.00358137}}]
}, Open  ]],

Cell["\<\
The setting n == 10 used above did not give a very accurate \
solution. When NDSolve computes the solution, it uses spatial error estimates \
on the initial condition to determine what the grid spacing should be. The \
error in the temporal (or at least time-like) variable is handled by the \
adaptive ODE integrator. \
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell["\<\
In the example (1), the distinction between time and space was \
quite clear from the problem context. Even when the distinction is not \
explicit, this document will refer to \"spatial\" and \"temporal\" variables. \
The \"spatial\" variables are those to which the discretization is done. The \
\"temporal\" variable is the one left in the ODE system to be \
integrated.\
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {
            StyleBox["TemporalVariable",
              "MR"], 
            StyleBox["Automatic",
              "MR"], 
            Cell["\<\
what variable to keep derivatives with respect to the \
derived ODE or DAE system\
\>", "Text"]},
          {
            StyleBox["Method",
              "MR"], 
            StyleBox["Automatic",
              "MR"], Cell[
            "what method to use for integrating the ODEs or DAEs", "Text"]},
          {
            StyleBox["SpatialDiscretization",
              "MR"], 
            StyleBox["TensorProductGrid",
              "MR"], Cell[
            "what method to use for spatial discretization ", "Text"]},
          {
            StyleBox["DifferentiateBoundaryConditions",
              "MR"], 
            StyleBox["True",
              "MR"], 
            Cell["\<\
whether to differentiate the boundary conditions with \
respect to the temporal variable\
\>"]},
          {
            StyleBox["ExpandFunctionSymbolically",
              "MR"], "\<\"\!\(\*
StyleBox[\\\"False\\\",\\n\\\"MR\\\"]\)\"\>", Cell[
            "whether to expand the effective function symbolically or not"]},
          {
            StyleBox["DiscretizedMonitorVariables",
              
              "MR"], "\<\"\\!\\(\\* \
StyleBox[\\\"False\\\",\\n\\\"MR\\\"]\\)\"\>", 
            RowBox[{"whether", " ", "to", " ", 
              ButtonBox["interpret",
                ButtonData:>"b:7.1.5",
                ButtonStyle->"Hyperlink"], " ", "dependent", " ", "variables",
               " ", "given", " ", "in", " ", "monitors", " ", "like", " ", 
              StyleBox["StepMonitor",
                "MR"], " ", "or", " ", "in", " ", "method", " ", "options", 
              " ", "for", " ", "methods", " ", "like", " ", 
              StyleBox["EventLocator",
                "MR"], " ", "and", " ", 
              StyleBox["Projection",
                "MR"], " ", "as", " ", "functions", " ", "of", " ", "the", 
              " ", "spatial", " ", "variables", " ", "or", " ", "vectors", 
              " ", "representing", " ", "the", " ", "spatially", " ", 
              "discretized", " ", \(\(values\)\(.\)\)}]}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell["Options for NDSolve`MethodOfLines. ", "Caption"],

Cell["\<\
Use of some of these options requires further knowledge of how the \
method of lines works and will be explained in the sections that \
follow.\
\>", "Text",
  CellTags->{"b:7.0", "ndsg:2.0.0"}],

Cell[TextData[{
  "Currently, the only method implemented for spatial discretization is the \
",
  StyleBox["TensorProductGrid", "MR"],
  " method, which uses discretization methods for one spatial dimension and \
uses an outer tensor product to derive methods for multiple spatial \
dimensions on rectangular regions.",
  " ",
  StyleBox["TensorProductGrid", "MR"],
  " has its own set of options that you can use to control the grid selection \
process. The following sections give sufficient background information so \
that you will be able to use these options if necessary."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Spatial Derivative Approximations", "Subsection",
  CellTags->{"c:3", "PDE:1.2"}],

Cell[CellGroupData[{

Cell["Finite Differences", "Subsubsection",
  CellTags->{"c:4", "b:7.1.0", "ndsg:2.0.1.0", "PDE:1.2.1"}],

Cell["\<\
The essence of the concept of finite differences is embodied in the \
standard definition of the derivative\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f' \((x\_i)\)\  == 
      lim\+\(h \[Rule] 0\)\[ThinSpace]\(f(h + x\_i) - f(x\_i)\)\/h\)], \
"DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "where instead of passing to the limit as h approaches zero, the finite \
spacing to the next adjacent point, ",
  Cell[BoxData[
      \(TraditionalForm\`x\_\(i + 1\) \[Equal] x\_i + h\)]],
  ", is used so that we get an approximation."
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f' \((x\_i)\)\_approx\  == \[ThinSpace]\(f(x\_\(i + \
1\)) - f(x\_i)\)\/h\)], "DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
The difference formula can also be derived from Taylor's formula, \
\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f(x\_\(i + 1\))\  \[Equal] \ 
      f(x\_i)\  + \ 
        h\ f' \((x\_i)\) + \(\(\(\ \)\(h\^2\)\)\/2\) f'' \((\[Xi]\_i)\); \ 
    x\_\(\(i\)\(\ \)\) < \ \[Xi]\_i\  < \ x\_\(i + 1\)\)], "DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
which is more useful since it provides an error estimate (assuming \
sufficient smoothness)\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f' \((x\_i)\)\  \[Equal] \ \[ThinSpace]\(f(x\_\(i + \
1\)) - f(x\_i)\)\/h - \ \(\(\(\(\ \)\(h\)\)\/2\) \(f''\) \((\[Xi]\_i)\)\(\ \)\
\)\)], "DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "An important aspect of this formula is that ",
  Cell[BoxData[
      \(TraditionalForm\`\[Xi]\_i\)]],
  "must lie between ",
  Cell[BoxData[
      \(TraditionalForm\`x\_i\)]],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`x\_\(i + 1\)\)]],
  "so that the error is local to the interval enclosing the sampling points. \
It is generally true for finite difference formulas that the error is local \
to the stencil, or set of sample points.",
  " ",
  "Typically, for convergence and other analysis, the error is expressed in \
asymptotic form:"
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f' \((x\_i)\)\ \  \[Equal] \ \[ThinSpace]\(\(\(f(
                x\_\(i + 1\)) - f(x\_i)\)\/h\)\(+\)\(O(
          h)\)\(\ \)\)\)], "DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "This formula is most commonly referred to as the first order forward \
difference.",
  " ",
  "The backward difference would use ",
  Cell[BoxData[
      \(TraditionalForm\`x\_\(i - 1\)\)]],
  "."
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
Taylor's formula can easily be used to derive higher order \
approximations. For example, subtracting \
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f(x\_\(i + 1\))\  \[Equal] \ 
      f(x\_i)\  + \ 
        h\ f' \((x\_i)\)\  + \(\(\(\ \)\(h\^2\)\)\/2\) f'' \((x\_i)\)\  + \ 
        O(h\^3)\)], "DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["from", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f(x\_\(i - 1\))\  \[Equal] \ 
      f(x\_i)\  - \ 
        h\ f' \((x\_i)\)\  + \(\(\(\ \)\(h\^2\)\)\/2\) f'' \((x\_i)\)\  + \ 
        O(h\^3)\)], "DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "and solving for ",
  Cell[BoxData[
      \(TraditionalForm\`f' \((x\_i)\)\)]],
  "gives the second order centered difference formula for the first \
derivative,"
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f' \((x\_i)\)\ \  \[Equal] \ \[ThinSpace]\(\(\(f(
                x\_\(i + 1\)) - f(x\_\(i - 1\))\)\/\(2  h\)\)\(+\)\(O(
          h\^2)\)\(\ \)\)\)], "DisplayMath",
  FontWeight->"Plain",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
If the Taylor formulas above are expanded out one order farther and \
added and then combined with the formula just above, it is not difficult to \
derive a centered formula for the second derivative.\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(TraditionalForm\`f'' \((x\_i)\)\ \  \[Equal] \ \[ThinSpace]\(\(\(f(
                x\_\(i + 1\)) - 2\ \(f(x\_i)\)\  + \ 
              f(x\_\(i - 1\))\)\/h\^2\)\(+\)\(O(
          h\^2)\)\(\ \)\)\)], "DisplayMath",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
Note that the while having a uniform step size h between points \
makes it convenient to write out the formulas, it is certainly not a \
requirement. For example, the approximation to the second derivative is in \
general\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{
          SuperscriptBox["f", "\[Prime]\[Prime]",
            MultilineFunction->None], "(", \(x\_i\), ")"}], 
        "==", \(\(2\ \((\(f(x\_\(i + 1\))\)\ \((x\_\(i - 1\) - x\_i)\) + \
\(f(x\_\(i - 1\))\)\ \((x\_i - x\_\(i + 1\))\) + \(f(x\_i)\)\ \((x\_\(i + 1\) \
- x\_\(i - 1\))\))\)\)\/\(\((x\_\(i - 1\) - x\_i)\)\ \((x\_\(i - 1\) - x\_\(i \
+ 1\))\)\ \((x\_i - x\_\(i + 1\))\)\) + O(h)\)}], 
      TraditionalForm]], "DisplayMath",
  CellLabel->"Out[65]=",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
where h corresponds to the maximum local grid spacing. Note that \
the asymptotic order of the three-point formula has dropped to first order; \
that it was second order on a uniform grid is due to fortuitous \
cancellations.\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
In general, formulas for any given derivative with asymptotic error \
of any chosen order can be derived from the Taylor formulas as long as a \
sufficient number of sample points are used. However, this method becomes \
cumbersome and inefficient beyond the simple examples shown above. An \
alternate formulation is based on polynomial interpolation: since the Taylor \
formulas are exact (no error term) for polynomials of sufficiently low order, \
so are the finite difference formulas. It is not difficult to show that the \
finite difference formulas are equivalent to the derivatives of interpolating \
polynomials. For example, a simple way of deriving the formula shown above \
for the second derivative is to interpolate a quadratic and find its second \
derivative (which is essentially just the leading coefficient).\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "This finds the three point finite difference formula for the second \
derivative by differentiating the polynomial interpolating the three points \
",
  Cell[BoxData[
      \(TraditionalForm\`\((x\_\(i - 1\), \ f(x\_\(i - 1\)))\)\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`\((x\_i, \ f(x\_i))\)\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`\((x\_\(i + 1\), \ f(x\_\(i + 1\)))\)\)]]
}], "MathCaption",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(D[InterpolatingPolynomial[
        Table[{\ x\_\(i + k\), f[x\_\(i + k\)]}, {k, \(-1\), 1}], z], z, 
      z]\)], "Input",
  CellLabel->"In[11]:=",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(\(2\ \((\(-\(\(\(-f[x\_\(\(-1\) + i\)]\) + f[x\_i]\)\/\(\(-x\_\(\(-1\) \
+ i\)\) + x\_i\)\)\) + \(\(-f[x\_i]\) + f[x\_\(1 + i\)]\)\/\(\(-x\_i\) + \
x\_\(1 + i\)\))\)\)\/\(\(-x\_\(\(-1\) + i\)\) + x\_\(1 + i\)\)\)], "Output",
  CellLabel->"Out[11]="]
}, Open  ]],

Cell[TextData[{
  "In this form of the formula, it is easy to see that it is effectively a \
difference of the forward and backward first order derivative approximations. \
Sometimes it is advantageous to use finite differences in this way, \
particularly for terms with coefficients inside of derivatives, such as (",
  Cell[BoxData[
      \(TraditionalForm\`\(\(\(a(x)\)\ u\_x\)\()\)\)\_x\)]],
  ", which commonly appear in PDEs.",
  " "
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
Another property made apparent by considering interpolation \
formulas is that the point at which you get the derivative approximation need \
not be on the grid. A common use of this is with staggered grids where the \
derivative may be wanted at the midpoints between grid points. \
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "This generates a fourth order approximation for the first derivative on a \
uniform staggered grid, ",
  Cell[BoxData[
      \(TraditionalForm\`x\_i\)]],
  ", where the main grid points ",
  Cell[BoxData[
      \(TraditionalForm\`x\_\(i + k/2\)\)]],
  "are at ",
  Cell[BoxData[
      \(TraditionalForm\`x\_i\)]],
  "+h k/2,",
  " ",
  "for odd k. "
}], "MathCaption",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(Simplify[
      D[InterpolatingPolynomial[
            Table[{\ x\_i + k\ h/2, f[x\_\(i + k/2\)]}, {k, \(-3\), 3, 2}], 
            z], z]\  /. \ z \[Rule] x\_i]\)], "Input",
  CellLabel->"In[12]:=",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(\(f[x\_\(\(-\(3\/2\)\) + i\)] - 27\ f[x\_\(\(-\(1\/2\)\) + i\)] + 27\ \
f[x\_\(1\/2 + i\)] - f[x\_\(3\/2 + i\)]\)\/\(24\ h\)\)], "Output",
  CellLabel->"Out[12]="]
}, Open  ]],

Cell[TextData[{
  "The fourth order error coefficient for this formula is ",
  Cell[BoxData[
      FormBox[
        RowBox[{\(3\/640\), " ", \(h\^4\), " ", 
          RowBox[{
            SuperscriptBox["f", 
              TagBox[\((5)\),
                Derivative],
              MultilineFunction->None], "(", \(x\_i\), ")"}]}], 
        TraditionalForm]]],
  " versus",
  Cell[BoxData[
      FormBox[
        RowBox[{\(1\/30\), " ", \(h\^4\), " ", 
          RowBox[{
            SuperscriptBox["f", 
              TagBox[\((5)\),
                Derivative],
              MultilineFunction->None], "(", \(x\_i\), ")"}]}], 
        TraditionalForm]]],
  " for the standard fourth order formula derived below.",
  " ",
  "Much of the reduced error can be attributed to the reduced stencil size."
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
This generates a fourth order approximation for the first \
derivative at a point on a uniform grid.\
\>", "MathCaption",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(Simplify[
      D[InterpolatingPolynomial[
            Table[{\ x\_i + k\ h, f[x\_\(i + k\)]}, {k, \(-2\), 2, 1}], z], 
          z]\  /. \ z \[Rule] x\_i]\)], "Input",
  CellLabel->"In[13]:=",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(\(f[x\_\(\(-2\) + i\)] - 8\ f[x\_\(\(-1\) + i\)] + 8\ f[x\_\(1 + i\)] - \
f[x\_\(2 + i\)]\)\/\(12\ h\)\)], "Output",
  CellLabel->"Out[13]="]
}, Open  ]],

Cell["\<\
In general, a finite difference formula using n points will be \
exact for functions that are polynomials of degree n - 1 and have asymptotic \
order at least n - m. On uniform grids, you can expect higher asymptotic \
order, especially for centered differences.\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "Using efficient polynomial interpolation techniques is a reasonable way to \
generate coefficients, but B. Fornberg has developed a fast algorithm for \
finite difference weight generation [",
  ButtonBox["F92",
    ButtonData:>{"References.nb", "F92"},
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["F98",
    ButtonData:>{"References.nb", "F98"},
    ButtonStyle->"Hyperlink"],
  "], which is substantially faster.",
  " "
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "In [",
  ButtonBox["F98",
    ButtonData:>{"References.nb", "F98"},
    ButtonStyle->"Hyperlink"],
  "], he presents a one-line ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " formula for explicit finite differences."
}], "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
This is the simple formula of Fornberg for generating weights on a \
uniform grid. Here it has been modified slightly by making it a function \
definition.\
\>", "MathCaption",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \(UFDWeights[m_, \ n_, \ s_]\  := \ 
      CoefficientList[Normal[Series[\(x\^s\) Log[x]\^m, {x, 1, n}]/h\^m], 
        x]\)], "Input",
  CellLabel->"In[14]:=",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
Here m is the order of the derivative, n is the number of grid \
intervals enclosed in the stencil, and s is the number of grid intervals \
between the point at which the derivative is approximated and the leftmost \
edge of the stencil. There is no requirement that s be an integer; noninteger \
values simply lead to staggered grid approximations. Setting s to be n/2 \
always generates a centered formula. \
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[{
  "This uses the Fornberg formula to generate the weights for a staggered \
fourth order approximation to the first derivative.",
  " ",
  "This is the same one computed above with ",
  StyleBox["InterpolatingPolynomial", "Input"],
  "."
}], "MathCaption",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(UFDWeights[1, \ 3, 3/2]\)], "Input",
  CellLabel->"In[15]:=",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[
    \({1\/\(24\ h\), \(-\(9\/\(8\ h\)\)\), 
      9\/\(8\ h\), \(-\(1\/\(24\ h\)\)\)}\)], "Output",
  CellLabel->"Out[15]="]
}, Open  ]],

Cell["\<\
A table of some commonly used finite difference formulas is given \
below for reference.\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[BoxData[{
    \(\(FDFormula[m_, \ n_, \ s_]\  := \ 
        Block[{}, 
          F = Table[
              f[Subscript[x, i + k]], {k, \(-s\), 
                n - s}]; \[IndentingNewLine]W = 
            PadRight[
              CoefficientList[Normal[Series[x^s\ Log[x]^m, {x, 1, n}]/h^m], 
                x], \ Length[F], \ 0]; \[IndentingNewLine]Print[
            W]; \[IndentingNewLine]Wfact = 
            1/Apply[PolynomialGCD, W]; \[IndentingNewLine]W = 
            Simplify[W\ Wfact]; \[IndentingNewLine]Print[{F, 
              W}]; \[IndentingNewLine]taylor[h_] = 
            Normal[Series[
                f[Subscript[x, i] + h], {h, 0, 
                  n + 2}]]; \[IndentingNewLine]error = 
            Drop[CoefficientList[
                Expand[\((Table[taylor[h\ k], {k, \(-s\), n - s}] . W)\)/
                    Wfact], h], 
              1]; \[IndentingNewLine]do = \(Position[error, 
                e_ /; e \[NotEqual] 0]\)[\([1, 
                1]\)]; \[IndentingNewLine]error = 
            error[\([do]\)]; \[IndentingNewLine]error = 
            error /. f_[Subscript[x, i]] \[Rule] 
                f; \[IndentingNewLine]error = 
            h^do\ error; \[IndentingNewLine]{\(\(Derivative[m]\)[f]\)[
                x\_i]\  \[TildeEqual] \ \((F . W)\)/Wfact, 
            error}];\)\), "\[IndentingNewLine]", 
    \(TableForm[
      Flatten[Table[
          Join[\[IndentingNewLine]Table[
              FDFormula[2, n, \ s], {s, n, \ Ceiling[n/2], \ \(-\ 1\)}], 
            Table[FDFormula[2, n, \ s], {s, Floor[n/2]\ \  - \ 1, 
                0, \ \(-\ 1\)}]], \[IndentingNewLine]{n, 3, 7, 2}], 1], \ 
      TableAlignments \[Rule] Center, \ 
      TableHeadings \[Rule] {{}, {"\<Formula\>", \ "\<Error Term\>"}}]\)}], \
"Input",
  CellLabel->"In[74]:=",
  Evaluatable->False,
  CellOpen->False,
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[Cell[BoxData[
    FormBox[GridBox[{
          {"Formula", \(Error\ term\)},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(f(x\_\(i - 2\)) - 4\ \(f(x\_\(i - 1\))\) + 
                    3\ \(f(x\_i)\)\)\/\(2\ h\)\)}], 
            RowBox[{\(1\/3\), " ", \(h\^2\), " ", 
              SuperscriptBox["f", 
                TagBox[\((3)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(f(x\_\(i + 1\)) - 
                    f(x\_\(i - 1\))\)\/\(2\ h\)\)}], 
            RowBox[{\(1\/6\), " ", \(h\^2\), " ", 
              SuperscriptBox["f", 
                TagBox[\((3)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-3\)\ \(f(x\_i)\) + 
                    4\ \(f(x\_\(i + 1\))\) - f(x\_\(i + 2\))\)\/\(2\ h\)\)}], 
            
            RowBox[{\(1\/3\), " ", \(h\^2\), " ", 
              SuperscriptBox["f", 
                TagBox[\((3)\),
                  Derivative],
                MultilineFunction->None]}]},
          {" ", " "},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(3\ \(f(x\_\(i - 4\))\) - 
                    16\ \(f(x\_\(i - 3\))\) + 36\ \(f(x\_\(i - 2\))\) - 
                    48\ \(f(x\_\(i - 1\))\) + 
                    25\ \(f(x\_i)\)\)\/\(12\ h\)\)}], 
            RowBox[{\(1\/5\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((5)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-\(f(x\_\(i - 3\))\)\) + 
                    6\ \(f(x\_\(i - 2\))\) - 18\ \(f(x\_\(i - 1\))\) + 
                    10\ \(f(x\_i)\) + 3\ \(f(x\_\(i + 1\))\)\)\/\(12\ h\)\)}],
             
            RowBox[{\(1\/20\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((5)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(f(x\_\(i - 2\)) - 8\ \(f(x\_\(i - 1\))\) + 
                    8\ \(f(x\_\(i + 1\))\) - f(x\_\(i + 2\))\)\/\(12\ h\)\)}],
             
            RowBox[{\(1\/30\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((5)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-3\)\ \(f(x\_\(i - 1\))\) - 
                    10\ \(f(x\_i)\) + 18\ \(f(x\_\(i + 1\))\) - 
                    6\ \(f(x\_\(i + 2\))\) + f(x\_\(i + 3\))\)\/\(12\ h\)\)}],
             
            RowBox[{\(1\/20\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((5)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-25\)\ \(f(x\_i)\) + 
                    48\ \(f(x\_\(i + 1\))\) - 36\ \(f(x\_\(i + 2\))\) + 
                    16\ \(f(x\_\(i + 3\))\) - 
                    3\ \(f(x\_\(i + 4\))\)\)\/\(12\ h\)\)}], 
            RowBox[{\(1\/5\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((5)\),
                  Derivative],
                MultilineFunction->None]}]},
          {" ", " "},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(10\ \(f(x\_\(i - 6\))\) - 
                    72\ \(f(x\_\(i - 5\))\) + 225\ \(f(x\_\(i - 4\))\) - 
                    400\ \(f(x\_\(i - 3\))\) + 450\ \(f(x\_\(i - 2\))\) - 
                    360\ \(f(x\_\(i - 1\))\) + 
                    147\ \(f(x\_i)\)\)\/\(60\ h\)\)}], 
            RowBox[{\(1\/7\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((7)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-2\)\ \(f(x\_\(i - 5\))\) + 
                    15\ \(f(x\_\(i - 4\))\) - 50\ \(f(x\_\(i - 3\))\) + 
                    100\ \(f(x\_\(i - 2\))\) - 150\ \(f(x\_\(i - 1\))\) + 
                    77\ \(f(x\_i)\) + 
                    10\ \(f(x\_\(i + 1\))\)\)\/\(60\ h\)\)}], 
            RowBox[{\(1\/42\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((7)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(f(x\_\(i - 4\)) - 8\ \(f(x\_\(i - 3\))\) + 
                    30\ \(f(x\_\(i - 2\))\) - 80\ \(f(x\_\(i - 1\))\) + 
                    35\ \(f(x\_i)\) + 24\ \(f(x\_\(i + 1\))\) - 
                    2\ \(f(x\_\(i + 2\))\)\)\/\(60\ h\)\)}], 
            RowBox[{\(1\/105\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((7)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-\(f(x\_\(i - 3\))\)\) + 
                    9\ \(f(x\_\(i - 2\))\) - 45\ \(f(x\_\(i - 1\))\) + 
                    45\ \(f(x\_\(i + 1\))\) - 9\ \(f(x\_\(i + 2\))\) + 
                    f(x\_\(i + 3\))\)\/\(60\ h\)\)}], 
            RowBox[{\(1\/140\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((7)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(2\ \(f(x\_\(i - 2\))\) - 
                    24\ \(f(x\_\(i - 1\))\) - 35\ \(f(x\_i)\) + 
                    80\ \(f(x\_\(i + 1\))\) - 30\ \(f(x\_\(i + 2\))\) + 
                    8\ \(f(x\_\(i + 3\))\) - f(x\_\(i + 4\))\)\/\(60\ h\)\)}],
             
            RowBox[{\(1\/105\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((7)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-10\)\ \(f(x\_\(i - 1\))\) - 
                    77\ \(f(x\_i)\) + 150\ \(f(x\_\(i + 1\))\) - 
                    100\ \(f(x\_\(i + 2\))\) + 50\ \(f(x\_\(i + 3\))\) - 
                    15\ \(f(x\_\(i + 4\))\) + 
                    2\ \(f(x\_\(i + 5\))\)\)\/\(60\ h\)\)}], 
            RowBox[{\(1\/42\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((7)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-147\)\ \(f(x\_i)\) + 
                    360\ \(f(x\_\(i + 1\))\) - 450\ \(f(x\_\(i + 2\))\) + 
                    400\ \(f(x\_\(i + 3\))\) - 225\ \(f(x\_\(i + 4\))\) + 
                    72\ \(f(x\_\(i + 5\))\) - 
                    10\ \(f(x\_\(i + 6\))\)\)\/\(60\ h\)\)}], 
            RowBox[{\(1\/7\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((7)\),
                  Derivative],
                MultilineFunction->None]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Center}], TraditionalForm]]]], "FramedBox",
  Background->RGBColor[0.964706, 0.929412, 0.839216],
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
Finite difference formulas on uniform grids for the first \
derivative.\
\>", "Caption",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell[TextData[Cell[BoxData[
    FormBox[GridBox[{
          {"Formula", \(Error\ Term\)},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-\(f(x\_\(i - 3\))\)\) + 
                    4\ \(f(x\_\(i - 2\))\) - 5\ \(f(x\_\(i - 1\))\) + 
                    2\ \(f(x\_i)\)\)\/h\^2\)}], 
            RowBox[{\(11\/12\), "  ", \(h\^2\), " ", 
              SuperscriptBox["f", 
                TagBox[\((4)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(f(x\_\(i - 1\)) - 2\ \(f(x\_i)\) + 
                    f(x\_\(i + 1\))\)\/h\^2\)}], 
            RowBox[{\(1\/12\), " ", \(h\^2\), " ", 
              SuperscriptBox["f", 
                TagBox[\((4)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(2\ \(f(x\_i)\) - 5\ \(f(x\_\(i + 1\))\) + 
                    4\ \(f(x\_\(i + 2\))\) - f(x\_\(i + 3\))\)\/h\^2\)}], 
            RowBox[{\(11\/12\), " ", \(h\^2\), " ", 
              SuperscriptBox["f", 
                TagBox[\((4)\),
                  Derivative],
                MultilineFunction->None]}]},
          {" ", " "},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-10\)\ \(f(x\_\(i - 5\))\) + 
                    61\ \(f(x\_\(i - 4\))\) - 156\ \(f(x\_\(i - 3\))\) + 
                    214\ \(f(x\_\(i - 2\))\) - 154\ \(f(x\_\(i - 1\))\) + 
                    45\ \(f(x\_i)\)\)\/\(12\ h\^2\)\)}], 
            RowBox[{\(137\/180\), "  ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((6)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(f(x\_\(i - 4\)) - 6\ \(f(x\_\(i - 3\))\) + 
                    14\ \(f(x\_\(i - 2\))\) - 4\ \(f(x\_\(i - 1\))\) - 
                    15\ \(f(x\_i)\) + 
                    10\ \(f(x\_\(i + 1\))\)\)\/\(12\ h\^2\)\)}], 
            RowBox[{\(13\/180\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((6)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-\(f(x\_\(i - 2\))\)\) + 
                    16\ \(f(x\_\(i - 1\))\) - 30\ \(f(x\_i)\) + 
                    16\ \(f(x\_\(i + 1\))\) - 
                    f(x\_\(i + 2\))\)\/\(12\ h\^2\)\)}], 
            RowBox[{\(1\/90\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((6)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(10\ \(f(x\_\(i - 1\))\) - 
                    15\ \(f(x\_i)\) - 4\ \(f(x\_\(i + 1\))\) + 
                    14\ \(f(x\_\(i + 2\))\) - 6\ \(f(x\_\(i + 3\))\) + 
                    f(x\_\(i + 4\))\)\/\(12\ h\^2\)\)}], 
            RowBox[{\(13\/180\), " ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((6)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(45\ \(f(x\_i)\) - 
                    154\ \(f(x\_\(i + 1\))\) + 214\ \(f(x\_\(i + 2\))\) - 
                    156\ \(f(x\_\(i + 3\))\) + 61\ \(f(x\_\(i + 4\))\) - 
                    10\ \(f(x\_\(i + 5\))\)\)\/\(12\ h\^2\)\)}], 
            RowBox[{\(137\/180\), "  ", \(h\^4\), " ", 
              SuperscriptBox["f", 
                TagBox[\((6)\),
                  Derivative],
                MultilineFunction->None]}]},
          {" ", " "},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-126\)\ \(f(x\_\(i - 7\))\) + 
                    1019\ \(f(x\_\(i - 6\))\) - 3618\ \(f(x\_\(i - 5\))\) + 
                    7380\ \(f(x\_\(i - 4\))\) - 9490\ \(f(x\_\(i - 3\))\) + 
                    7911\ \(f(x\_\(i - 2\))\) - 4014\ \(f(x\_\(i - 1\))\) + 
                    938\ \(f(x\_i)\)\)\/\(180\ h\^2\)\)}], 
            RowBox[{\(363\/560\), "  ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((8)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(11\ \(f(x\_\(i - 6\))\) - 
                    90\ \(f(x\_\(i - 5\))\) + 324\ \(f(x\_\(i - 4\))\) - 
                    670\ \(f(x\_\(i - 3\))\) + 855\ \(f(x\_\(i - 2\))\) - 
                    486\ \(f(x\_\(i - 1\))\) - 70\ \(f(x\_i)\) + 
                    126\ \(f(x\_\(i + 1\))\)\)\/\(180\ h\^2\)\)}], 
            RowBox[{\(29\/560\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((8)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-2\)\ \(f(x\_\(i - 5\))\) + 
                    16\ \(f(x\_\(i - 4\))\) - 54\ \(f(x\_\(i - 3\))\) + 
                    85\ \(f(x\_\(i - 2\))\) + 130\ \(f(x\_\(i - 1\))\) - 
                    378\ \(f(x\_i)\) + 214\ \(f(x\_\(i + 1\))\) - 
                    11\ \(f(x\_\(i + 2\))\)\)\/\(180\ h\^2\)\)}], 
            RowBox[{\(47\/5040\), \(h\^6\), "  ", 
              SuperscriptBox["f", 
                TagBox[\((8)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(2\ \(f(x\_\(i - 3\))\) - 
                    27\ \(f(x\_\(i - 2\))\) + 270\ \(f(x\_\(i - 1\))\) - 
                    490\ \(f(x\_i)\) + 270\ \(f(x\_\(i + 1\))\) - 
                    27\ \(f(x\_\(i + 2\))\) + 
                    2\ \(f(x\_\(i + 3\))\)\)\/\(180\ h\^2\)\)}], 
            RowBox[{\(1\/560\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((8)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(\(-11\)\ \(f(x\_\(i - 2\))\) + 
                    214\ \(f(x\_\(i - 1\))\) - 378\ \(f(x\_i)\) + 
                    130\ \(f(x\_\(i + 1\))\) + 85\ \(f(x\_\(i + 2\))\) - 
                    54\ \(f(x\_\(i + 3\))\) + 16\ \(f(x\_\(i + 4\))\) - 
                    2\ \(f(x\_\(i + 5\))\)\)\/\(180\ h\^2\)\)}], 
            RowBox[{\(47\/5040\), \(h\^6\), "  ", 
              SuperscriptBox["f", 
                TagBox[\((8)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(126\ \(f(x\_\(i - 1\))\) - 
                    70\ \(f(x\_i)\) - 486\ \(f(x\_\(i + 1\))\) + 
                    855\ \(f(x\_\(i + 2\))\) - 670\ \(f(x\_\(i + 3\))\) + 
                    324\ \(f(x\_\(i + 4\))\) - 90\ \(f(x\_\(i + 5\))\) + 
                    11\ \(f(x\_\(i + 6\))\)\)\/\(180\ h\^2\)\)}], 
            RowBox[{\(29\/560\), " ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((8)\),
                  Derivative],
                MultilineFunction->None]}]},
          {
            RowBox[{
              RowBox[{
                SuperscriptBox["f", "\[Prime]\[Prime]",
                  MultilineFunction->None], "(", \(x\_i\), ")"}], 
              "\[TildeEqual]", \(\(938\ \(f(x\_i)\) - 
                    4014\ \(f(x\_\(i + 1\))\) + 7911\ \(f(x\_\(i + 2\))\) - 
                    9490\ \(f(x\_\(i + 3\))\) + 7380\ \(f(x\_\(i + 4\))\) - 
                    3618\ \(f(x\_\(i + 5\))\) + 1019\ \(f(x\_\(i + 6\))\) - 
                    126\ \(f(x\_\(i + 7\))\)\)\/\(180\ h\^2\)\)}], 
            RowBox[{\(363\/560\), "  ", \(h\^6\), " ", 
              SuperscriptBox["f", 
                TagBox[\((8)\),
                  Derivative],
                MultilineFunction->None]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Center}], TraditionalForm]]]], "FramedBox",
  Background->RGBColor[0.964706, 0.929412, 0.839216],
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
Finite difference formulas on uniform grids for the second \
derivative.\
\>", "Caption",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}],

Cell["\<\
One thing to notice from this table is that the farther the \
formulas get from centered, the larger the error term coefficient, sometimes \
by factors of hundreds. For this reason, sometimes where one-sided derivative \
formulas are required (such as at boundaries), formulas of higher order are \
used to offset the extra error.\
\>", "Text",
  CellTags->{"b:7.1.0", "ndsg:2.0.1.0"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["NDSolve`FiniteDifferenceDerivative", "Subsubsection",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1", "PDE:1.2.2"}],

Cell[TextData[{
  "Fornberg [",
  ButtonBox["F92",
    ButtonData:>{"References.nb", "F92"},
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["F98",
    ButtonData:>{"References.nb", "F98"},
    ButtonStyle->"Hyperlink"],
  "] also gives an algorithm that, though not quite so elegant and simple, is \
more general and, in particular, is applicable to nonuniform grids.",
  " ",
  "It is not difficult to program in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", but to make it as efficient as possible, a new kernel function has been \
provided as a simpler interface (along with some additional features)."
}], "Text",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[TextData[Cell[BoxData[
    FormBox[GridBox[{
          {
            StyleBox[
              RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
                RowBox[{
                  RowBox[{"Derivative", "[", 
                    StyleBox["m",
                      FontSlant->"Italic"], "]"}], ",", " ", 
                  StyleBox["grid",
                    FontSlant->"Italic"], ",", " ", 
                  StyleBox["values",
                    FontSlant->"Italic"]}], "]"}],
              FontFamily->"Courier"], 
            RowBox[{
            "approximate", " ", "the", " ", \(m\^th\), " ", "order", " ", 
              "derivative", " ", "for", " ", "the", " ", "function", " ", 
              "that", " ", "takes", " ", "on", " ", 
              StyleBox["values",
                FontFamily->"Courier",
                FontSlant->"Italic"], " ", "on", " ", "the", " ", 
              RowBox[{
                StyleBox["grid",
                  FontFamily->"Courier",
                  FontSlant->"Italic"], "."}]}]},
          {
            StyleBox[
              RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
                RowBox[{
                  RowBox[{"Derivative", "[", 
                    StyleBox[\(m1, \ m2, \  ... , \ mn\),
                      FontSlant->"Italic"], "]"}], ",", " ", 
                  RowBox[{"{", 
                    StyleBox[\(grid1, \ grid2, \  ... , \ gridn\),
                      FontSlant->"Italic"], 
                    StyleBox["}",
                      FontSlant->"Italic"]}], ",", " ", 
                  StyleBox["values",
                    FontSlant->"Italic"]}], "]"}],
              FontFamily->"Courier"], 
            RowBox[{
            "approximate", " ", "the", " ", "partial", " ", "derivative", " ",
               "of", " ", "order", " ", \((m1, \ m2, \  ... , \ mn)\), " ", 
              "for", " ", "the", " ", "function", " ", "of", " ", "n", " ", 
              "variables", " ", "which", " ", "takes", " ", "on", " ", 
              StyleBox["values",
                FontFamily->"Courier",
                FontSlant->"Italic"], " ", "on", " ", "the", " ", "tensor", 
              " ", "product", " ", "grid", " ", "defined", " ", "by", " ", 
              "the", " ", "outer", " ", "product", " ", "of", " ", 
              RowBox[{
                RowBox[{"(", " ", 
                  StyleBox[\(grid1, \ grid2, \  ... , \ gridn\),
                    FontFamily->"Courier",
                    FontSlant->"Italic"], 
                  StyleBox[")",
                    FontFamily->"Courier",
                    FontSlant->"Italic"]}], ".", "  "}]}]},
          {
            StyleBox[
              RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
                RowBox[{
                  RowBox[{"Derivative", "[", 
                    StyleBox[\(m1, \ m2, \  ... , \ mn\),
                      FontSlant->"Italic"], "]"}], ",", " ", 
                  RowBox[{"{", 
                    StyleBox[\(grid1, \ grid2, \  ... , \ gridn\),
                      FontSlant->"Italic"], 
                    StyleBox["}",
                      FontSlant->"Italic"]}]}], 
                StyleBox["]",
                  FontSlant->"Italic"]}],
              FontFamily->"Courier"], 
            RowBox[{
              RowBox[{
              "compute", " ", "the", " ", "finite", " ", "difference", " ", 
                "weights", " ", "needed", " ", "to", " ", "approximate", " ", 
                "the", " ", "partial", " ", "derivative", " ", "of", " ", 
                "order", " ", \((m1, \ m2, \  ... , \ mn)\), " ", "for", " ", 
                "the", " ", "function", " ", "of", " ", "n", " ", "variables",
                 " ", "on", " ", "the", " ", "tensor", " ", "product", " ", 
                "grid", " ", "defined", " ", "by", " ", "the", " ", "outer", 
                " ", "product", " ", "of", " ", 
                RowBox[{
                  RowBox[{"(", " ", 
                    StyleBox[\(grid1, \ grid2, \  ... , \ gridn\),
                      FontFamily->"Courier",
                      FontSlant->"Italic"], 
                    StyleBox[")",
                      FontFamily->"Courier",
                      FontSlant->"Italic"]}], ".", "  ", "The"}], " ", 
                "result", " ", "is", " ", "returned", " ", "as", " ", "an", 
                " ", 
                StyleBox["NDSolve`FiniteDifferenceDerivativeFunction",
                  FontFamily->"Courier"]}], 
              ",", \(which\ can\ be\ repeatedly\ applied\ to\ values\ on\ the\
\ \(\(grid\)\(.\)\)\)}]}
          }], TraditionalForm]]]], "DefinitionBox",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["Finding finite difference approximations to derivatives.", "Caption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[TextData[StyleBox["This defines a uniform grid with points spaced apart \
by a symbolic distance h.", "MathCaption"]], "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(ugrid\  = \ h\ Range[0, 8]\)], "Input",
  CellLabel->"In[16]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \({0, h, 2\ h, 3\ h, 4\ h, 5\ h, 6\ h, 7\ h, 8\ h}\)], "Output",
  CellLabel->"Out[16]="]
}, Open  ]],

Cell["\<\
This gives the first derivative formulas on the grid for a symbolic \
function f.\
\>", "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`FiniteDifferenceDerivative[Derivative[1], ugrid, \ 
      Map[f, \ ugrid]]\)], "Input",
  CellLabel->"In[17]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \({\(-\(\(25\ f[
                  0]\)\/\(12\ h\)\)\) + \(4\ f[h]\)\/h - \(3\ f[2\ h]\)\/h + \
\(4\ f[3\ h]\)\/\(3\ h\) - 
        f[4\ h]\/\(4\ h\), \(-\(f[
                0]\/\(4\ h\)\)\) - \(5\ f[h]\)\/\(6\ h\) + \(3\ f[2\ \
h]\)\/\(2\ h\) - f[3\ h]\/\(2\ h\) + f[4\ h]\/\(12\ h\), 
      f[0]\/\(12\ h\) - \(2\ f[h]\)\/\(3\ h\) + \(2\ f[3\ h]\)\/\(3\ h\) - 
        f[4\ h]\/\(12\ h\), 
      f[h]\/\(12\ h\) - \(2\ f[2\ h]\)\/\(3\ h\) + \(2\ f[4\ h]\)\/\(3\ h\) - 
        f[5\ h]\/\(12\ h\), 
      f[2\ h]\/\(12\ h\) - \(2\ f[3\ h]\)\/\(3\ h\) + \(2\ f[5\ h]\)\/\(3\ \
h\) - f[6\ h]\/\(12\ h\), 
      f[3\ h]\/\(12\ h\) - \(2\ f[4\ h]\)\/\(3\ h\) + \(2\ f[6\ h]\)\/\(3\ \
h\) - f[7\ h]\/\(12\ h\), 
      f[4\ h]\/\(12\ h\) - \(2\ f[5\ h]\)\/\(3\ h\) + \(2\ f[7\ h]\)\/\(3\ \
h\) - f[8\ h]\/\(12\ h\), \(-\(f[4\ h]\/\(12\ h\)\)\) + 
        f[5\ h]\/\(2\ h\) - \(3\ f[6\ h]\)\/\(2\ h\) + \(5\ f[7\ h]\)\/\(6\ h\
\) + f[8\ h]\/\(4\ h\), 
      f[4\ h]\/\(4\ h\) - \(4\ f[5\ h]\)\/\(3\ h\) + \(3\ f[6\ h]\)\/h - \(4\ \
f[7\ h]\)\/h + \(25\ f[8\ h]\)\/\(12\ h\)}\)], "Output",
  CellLabel->"Out[17]="]
}, Open  ]],

Cell["\<\
The derivatives at the endpoints are computed using one-sided \
formulas. The formulas shown in the example above are fourth order accurate, \
which is the default. In general, when you use symbolic grid and/or data, you \
will get symbolic formulas. This is often useful for doing analysis on the \
methods; however, for actual numerical grids, it is usually faster and more \
accurate to give the numerical grid to NDSolve`FiniteDifferenceDerivative \
rather than using the symbolic formulas.\
\>", "Text",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["\<\
This defines a randomly spaced grid between 0 and 2 \[Pi].\
\>", \
"MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(rgrid\  = \ 
      Sort[Join[{0, \ 2\ \[Pi]}, \ 
          Table[2  \[Pi]\ Random[], {10}]]]\)], "Input",
  CellLabel->"In[18]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \({0, 1.295318827690317`, 2.6440667528933304`, 2.7712954303341966`, 
      2.9194782932232988`, 3.316858670423088`, 4.165091214035327`, 
      4.541843384063749`, 5.003451989294785`, 5.242573608851487`, 
      5.340907400612432`, 2\ \[Pi]}\)], "Output",
  CellLabel->"Out[18]="]
}, Open  ]],

Cell["\<\
This approximates the derivative of the sine function at each point \
on the grid.\
\>", "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`FiniteDifferenceDerivative[Derivative[1], rgrid, \ 
      Sin[rgrid]]\)], "Input",
  CellLabel->"In[19]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \({1.0077584593555253`, 
      0.26391510173163674`, \(-0.8783201219513552`\), \
\(-0.9321130826184576`\), \(-0.975268635427687`\), \(-0.9829759598563446`\), \
\(-0.5186667134192805`\), \(-0.16942903234823478`\), 0.2869269871509373`, 
      0.5056178660405766`, 0.5880587294154599`, 
      0.9881236361167516`}\)], "Output",
  CellLabel->"Out[19]="]
}, Open  ]],

Cell["This shows the error in the approximations.", "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(%\  - \ Cos[rgrid]\)], "Input",
  CellLabel->"In[20]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \({0.007758459355525327`, \(-0.008091361420022258`\), 
      0.0004458992065780043`, 0.00010674129152266065`, 
      0.00016521679727354854`, 0.0017042285116569689`, 
      0.0017148895161849254`, 
      0.0002910229341778481`, \(-0.000043697978505730095`\), \
\(-0.00007476529370897822`\), 
      0.00011177485637148266`, \(-0.01187636388324842`\)}\)], "Output",
  CellLabel->"Out[20]="]
}, Open  ]],

Cell["\<\
In multiple dimensions, NDSolve`FiniteDifferenceDerivative works on \
tensor product grids, and you only need to specify the grid points for each \
dimension. \
\>", "Text",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["\<\
This defines grids xgrid and ygrid for the x and y direction, gives \
an approximation for the mixed xy partial derivative of the Gaussian on the \
tensor product of xgrid and ygrid, and makes a surface plot of the \
error.\
\>", "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(xgrid\  = \ Range[0, 8];\)\), "\[IndentingNewLine]", 
    \(\(ygrid\  = \ Range[0, 10];\)\), "\[IndentingNewLine]", 
    \(\(gaussian[x_, \ y_]\  = \ 
        Exp[\(-\((\((x\  - \ 4)\)\^2 + \((y\  - \ 5)\)\^2)\)\)/
            10];\)\), "\[IndentingNewLine]", 
    \(\(values\  = \ \ Outer[gaussian, xgrid, \ 
          ygrid];\)\), "\[IndentingNewLine]", 
    \(\(ListPlot3D[
        NDSolve`FiniteDifferenceDerivative[{1, 1}, {xgrid, \ ygrid}, 
            values]\  - \ 
          Outer[Function[{x, y}, Evaluate[D[gaussian[x, y], x, y]]], \ 
            xgrid, \ ygrid], \ 
        MeshRange \[Rule] {{0, 8}, {0, 10}}];\)\)}], "Input",
  CellLabel->"In[21]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
3.1984e-017 1.04977 -0.0679587 1.04977 [
[.02757 .24618 -6 -8.70276 ]
[.02757 .24618 0 .29724 ]
[.17646 .18758 -5.89944 -9 ]
[.17646 .18758 .10056 0 ]
[.3345 .12542 -5.5867 -9 ]
[.3345 .12542 .4133 0 ]
[.50253 .05935 -5.27396 -9 ]
[.50253 .05935 .72604 0 ]
[.68156 -0.01098 -4.96123 -9 ]
[.68156 -0.01098 1.03877 0 ]
[.70096 -0.00478 0 -6.26206 ]
[.70096 -0.00478 6 2.73794 ]
[.76745 .09703 0 -6.16187 ]
[.76745 .09703 6 2.83813 ]
[.82812 .18993 0 -6.07246 ]
[.82812 .18993 6 2.92754 ]
[.88369 .27502 0 -5.99218 ]
[.88369 .27502 6 3.00782 ]
[.93478 .35327 0 -5.9197 ]
[.93478 .35327 6 3.0803 ]
[.98191 .42546 0 -5.85393 ]
[.98191 .42546 12 3.14607 ]
[.01798 .30354 -36 -2.77479 ]
[.01798 .30354 0 6.22521 ]
[.00696 .37264 -6 -2.83339 ]
[.00696 .37264 0 6.16661 ]
[-0.00457 .44487 -30 -2.89492 ]
[-0.00457 .44487 0 6.10508 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.18558 .19702 m
.19014 .20174 L
s
[(2)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(4)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(6)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(8)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.07346 .24092 m
.0763 .24365 L
s
.11029 .2265 m
.1131 .22926 L
s
.14766 .21187 m
.15043 .21466 L
s
.22407 .18195 m
.22677 .18482 L
s
.26314 .16665 m
.2658 .16956 L
s
.3028 .15112 m
.30541 .15406 L
s
.38395 .11935 m
.38647 .12237 L
s
.42547 .10309 m
.42794 .10615 L
s
.46763 .08658 m
.47006 .08968 L
s
.55397 .05277 m
.55629 .05595 L
s
.59817 .03546 m
.60044 .03868 L
s
.64309 .01787 m
.6453 .02113 L
s
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.68874 0 m
.68263 .00239 L
s
[(0)] .70096 -0.00478 -1 .39157 Mshowa
.75514 .10158 m
.74899 .10385 L
s
[(2)] .76745 .09703 -1 .3693 Mshowa
.81573 .19425 m
.80953 .19642 L
s
[(4)] .82812 .18993 -1 .34944 Mshowa
.87123 .27915 m
.865 .28122 L
s
[(6)] .88369 .27502 -1 .3316 Mshowa
.92227 .35722 m
.91601 .35919 L
s
[(8)] .93478 .35327 -1 .31549 Mshowa
.96935 .42924 m
.96306 .43113 L
s
[(10)] .98191 .42546 -1 .30087 Mshowa
.125 Mabswid
.70593 .0263 m
.70226 .02771 L
s
.72272 .05198 m
.71904 .05338 L
s
.73912 .07706 m
.73543 .07845 L
s
.7708 .12553 m
.7671 .12688 L
s
.78611 .14895 m
.78241 .15028 L
s
.80108 .17185 m
.79737 .17317 L
s
.83006 .21617 m
.82633 .21745 L
s
.84407 .23761 m
.84035 .23888 L
s
.8578 .2586 m
.85407 .25986 L
s
.88439 .29928 m
.88065 .3005 L
s
.89727 .31899 m
.89353 .3202 L
s
.9099 .3383 m
.90615 .3395 L
s
.93439 .37576 m
.93063 .37693 L
s
.94627 .39394 m
.94251 .3951 L
s
.95792 .41176 m
.95416 .41291 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.03023 .29884 m
.03636 .29649 L
s
[(-0.002)] .01798 .30354 1 -0.38338 Mshowa
.01926 .36808 m
.02541 .3658 L
s
[(0)] .00696 .37264 1 -0.37036 Mshowa
.00779 .44046 m
.01397 .43826 L
s
[(0.002)] -0.00457 .44487 1 -0.35668 Mshowa
.125 Mabswid
.02754 .31587 m
.03122 .31447 L
s
.02481 .33308 m
.02849 .33169 L
s
.02205 .35048 m
.02574 .34911 L
s
.01644 .38587 m
.02014 .38452 L
s
.01359 .40387 m
.01729 .40252 L
s
.01071 .42206 m
.01441 .42073 L
s
.0329 .282 m
.03658 .28058 L
s
.03554 .26533 m
.03921 .26391 L
s
.00484 .45907 m
.00855 .45776 L
s
.00186 .4779 m
.00557 .4766 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
.59 .569 .799 r
.36287 .63993 .40113 .71789 .45365 .74262 .41599 .63608 Metetra
.655 .559 .734 r
.41599 .63608 .45365 .74262 .50851 .75666 .47089 .6118 Metetra
.58 .479 .705 r
.47089 .6118 .50851 .75666 .56531 .76183 .52665 .61092 Metetra
.592 .484 .7 r
.52665 .61092 .56531 .76183 .62257 .71666 .58402 .60538 Metetra
.64 .55 .737 r
.58402 .60538 .62257 .71666 .67796 .62396 .64248 .58525 Metetra
.677 .758 .912 r
.64248 .58525 .67796 .62396 .73209 .53338 .70208 .56472 Metetra
.323 0 0 r
.70208 .56472 .73209 .53338 .78848 .48871 .76387 .55834 Metetra
.489 .104 .237 r
.76387 .55834 .78848 .48871 .84958 .48957 .82786 .55656 Metetra
.339 0 .17 r
.82786 .55656 .84958 .48957 .9137 .49926 .89115 .52944 Metetra
.322 0 0 r
.89115 .52944 .9137 .49926 .98187 .52084 .95831 .52348 Metetra
.105 .576 .912 r
.32045 .61656 .36287 .63993 .41599 .63608 .37374 .62865 Metetra
.394 .827 .957 r
.37374 .62865 .41599 .63608 .47089 .6118 .42968 .61909 Metetra
.247 0 0 r
.42968 .61909 .47089 .6118 .52665 .61092 .48689 .62436 Metetra
.283 .771 .697 r
.48689 .62436 .52665 .61092 .58402 .60538 .54592 .60199 Metetra
.862 .96 .867 r
.54592 .60199 .58402 .60538 .64248 .58525 .60524 .5446 Metetra
.821 .786 .819 r
.60524 .5446 .64248 .58525 .70208 .56472 .66445 .48731 Metetra
.666 .626 .8 r
.66445 .48731 .70208 .56472 .76387 .55834 .72559 .46349 Metetra
.554 .534 .787 r
.72559 .46349 .76387 .55834 .82786 .55656 .79007 .46618 Metetra
.615 .579 .79 r
.79007 .46618 .82786 .55656 .89115 .52944 .85535 .45379 Metetra
.502 .554 .842 r
.85535 .45379 .89115 .52944 .95831 .52348 .92518 .46377 Metetra
.351 .677 .978 r
.2761 .58539 .32045 .61656 .37374 .62865 .33036 .59041 Metetra
.589 .738 .947 r
.33036 .59041 .37374 .62865 .42968 .61909 .38712 .5807 Metetra
.5 .707 .963 r
.38712 .5807 .42968 .61909 .48689 .62436 .4452 .57885 Metetra
.693 .743 .89 r
.4452 .57885 .48689 .62436 .54592 .60199 .50526 .55403 Metetra
.793 .759 .821 r
.50526 .55403 .54592 .60199 .60524 .5446 .56609 .50188 Metetra
.804 .785 .835 r
.56609 .50188 .60524 .5446 .66445 .48731 .62705 .44963 Metetra
.699 .793 .923 r
.62705 .44963 .66445 .48731 .72559 .46349 .68946 .42334 Metetra
.525 .718 .961 r
.68946 .42334 .72559 .46349 .79007 .46618 .75461 .41887 Metetra
.605 .721 .926 r
.75461 .41887 .79007 .46618 .85535 .45379 .82116 .40609 Metetra
.483 .663 .942 r
.82116 .40609 .85535 .45379 .92518 .46377 .89134 .40823 Metetra
.653 .784 .944 r
.22944 .55152 .2761 .58539 .33036 .59041 .28596 .53342 Metetra
.653 .684 .866 r
.28596 .53342 .33036 .59041 .38712 .5807 .34373 .51493 Metetra
.649 .656 .843 r
.34373 .51493 .38712 .5807 .4452 .57885 .40277 .49602 Metetra
.641 .611 .804 r
.40277 .49602 .4452 .57885 .50526 .55403 .46314 .4767 Metetra
.645 .628 .819 r
.46314 .4767 .50526 .55403 .56609 .50188 .52489 .45693 Metetra
.656 .751 .92 r
.52489 .45693 .56609 .50188 .62705 .44963 .58805 .43671 Metetra
.56 .896 .983 r
.58805 .43671 .62705 .44963 .68946 .42334 .65268 .41602 Metetra
.498 .893 .948 r
.65268 .41602 .68946 .42334 .75461 .41887 .71882 .39484 Metetra
.616 .868 .99 r
.71882 .39484 .75461 .41887 .82116 .40609 .78655 .37316 Metetra
.64 .832 .974 r
.78655 .37316 .82116 .40609 .89134 .40823 .8559 .35095 Metetra
.777 .797 .869 r
.18022 .51579 .22944 .55152 .28596 .53342 .23973 .47407 Metetra
.692 .696 .849 r
.23973 .47407 .28596 .53342 .34373 .51493 .29872 .44672 Metetra
.715 .68 .815 r
.29872 .44672 .34373 .51493 .40277 .49602 .35918 .41091 Metetra
.612 .597 .812 r
.35918 .41091 .40277 .49602 .46314 .4767 .41968 .39688 Metetra
.438 .524 .848 r
.41968 .39688 .46314 .4767 .52489 .45693 .48147 .40955 Metetra
.308 .59 .943 r
.48147 .40955 .52489 .45693 .58805 .43671 .54606 .42279 Metetra
.46 .852 .978 r
.54606 .42279 .58805 .43671 .65268 .41602 .61288 .40809 Metetra
.698 .965 .938 r
.61288 .40809 .65268 .41602 .71882 .39484 .68046 .36908 Metetra
.689 .892 .971 r
.68046 .36908 .71882 .39484 .78655 .37316 .74959 .338 Metetra
.789 .873 .91 r
.74959 .338 .78655 .37316 .8559 .35095 .81859 .29067 Metetra
.8 .8 .851 r
.12809 .47909 .18022 .51579 .23973 .47407 .18979 .42909 Metetra
.702 .764 .9 r
.18979 .42909 .23973 .47407 .29872 .44672 .24986 .40119 Metetra
.769 .776 .859 r
.24986 .40119 .29872 .44672 .35918 .41091 .31207 .35763 Metetra
.625 .714 .91 r
.31207 .35763 .35918 .41091 .41968 .39688 .37353 .34105 Metetra
.254 .526 .914 r
.37353 .34105 .41968 .39688 .48147 .40955 .43565 .35956 Metetra
.206 .53 .922 r
.43565 .35956 .48147 .40955 .54606 .42279 .50095 .37901 Metetra
.608 .772 .959 r
.50095 .37901 .54606 .42279 .61288 .40809 .56915 .36163 Metetra
.777 .814 .882 r
.56915 .36163 .61288 .40809 .68046 .36908 .63825 .31415 Metetra
.697 .763 .903 r
.63825 .31415 .68046 .36908 .74959 .338 .70898 .28226 Metetra
.797 .796 .851 r
.70898 .28226 .74959 .338 .81859 .29067 .77919 .22604 Metetra
.739 .832 .922 r
.07194 .44715 .12809 .47909 .18979 .42909 .13335 .41294 Metetra
.466 .84 .989 r
.13335 .41294 .18979 .42909 .24986 .40119 .19344 .40044 Metetra
.716 .968 .936 r
.19344 .40044 .24986 .40119 .31207 .35763 .25753 .36132 Metetra
.625 .957 .914 r
.25753 .36132 .31207 .35763 .37353 .34105 .32217 .32662 Metetra
.537 .884 .985 r
.32217 .32662 .37353 .34105 .43565 .35956 .38723 .30672 Metetra
.633 .744 .928 r
.38723 .30672 .43565 .35956 .50095 .37901 .45405 .28627 Metetra
.685 .643 .802 r
.45405 .28627 .50095 .37901 .56915 .36163 .52272 .24967 Metetra
.685 .609 .765 r
.52272 .24967 .56915 .36163 .63825 .31415 .59248 .20745 Metetra
.596 .574 .8 r
.59248 .20745 .63825 .31415 .70898 .28226 .66444 .19073 Metetra
.695 .661 .813 r
.66444 .19073 .70898 .28226 .77919 .22604 .73712 .15087 Metetra
.755 .65 .745 r
.02204 .35054 .07194 .44715 .13335 .41294 .0879 .28633 Metetra
.701 .586 .723 r
.0879 .28633 .13335 .41294 .19344 .40044 .15215 .23389 Metetra
.65 .519 .69 r
.15215 .23389 .19344 .40044 .25753 .36132 .21579 .19024 Metetra
.494 .416 .697 r
.21579 .19024 .25753 .36132 .32217 .32662 .27607 .19512 Metetra
.122 .229 .703 r
.27607 .19512 .32217 .32662 .38723 .30672 .33597 .25079 Metetra
0 .063 .577 r
.33597 .25079 .38723 .30672 .45405 .28627 .40081 .31104 Metetra
.307 0 0 r
.40081 .31104 .45405 .28627 .52272 .24967 .47218 .31941 Metetra
.47218 .31941 .52272 .24967 .59248 .20745 .54672 .27015 Metetra
.54672 .27015 .59248 .20745 .66444 .19073 .62185 .20982 Metetra
.702 .96 .705 r
.62185 .20982 .66444 .19073 .73712 .15087 .69664 .13618 Metetra
0 g
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.18558 .19702 m
.19014 .20174 L
s
[(2)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(4)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(6)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(8)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.07346 .24092 m
.0763 .24365 L
s
.11029 .2265 m
.1131 .22926 L
s
.14766 .21187 m
.15043 .21466 L
s
.22407 .18195 m
.22677 .18482 L
s
.26314 .16665 m
.2658 .16956 L
s
.3028 .15112 m
.30541 .15406 L
s
.38395 .11935 m
.38647 .12237 L
s
.42547 .10309 m
.42794 .10615 L
s
.46763 .08658 m
.47006 .08968 L
s
.55397 .05277 m
.55629 .05595 L
s
.59817 .03546 m
.60044 .03868 L
s
.64309 .01787 m
.6453 .02113 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[21]:=",
  ImageSize->{288, 233.562},
  ImageMargins->{{2, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jB000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00a@3oool2000005T0oooo00340?ooo`040000003oool0oooo000005P0oooo00340?ooo`04
0000003oool0oooo000005P0oooo00350?ooo`800000F@3oool00<@0oooo00@000000?ooo`3oool0
0000F03oool00<D0oooo0P0000080?ooo`800000C`3oool00<h0oooo00@000000?ooo`3oool00000
CP3oool00<h0oooo00@000000?ooo`3oool00000CP3oool00<h0oooo00@000000?ooo`3oool00000
CP3oool00<h0oooo00@000000?ooo`3oool00000CP3oool00<l0oooo0P00001?0?ooo`00b03oool3
000005D0oooo00370?ooo`80000000<0oooo000000000000E03oool00<@0oooo0`0000030?ooo`80
0000E03oool00<80oooo0P0000060?ooo`030000003oool0000005<0oooo002n0?ooo`@00000203o
ool010000000oooo0?ooo`00001B0?ooo`00_03oool4000000X0oooo00@000000?ooo`3oool00000
DP3oool009L0oooo0`00000P0?ooo`8000003P3oool00`000000oooo0?ooo`0200000540oooo002G
0?ooo`040000003oool0oooo000001`0oooo0`00000@0?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool0CP3oool009L0oooo00@000000?ooo`3oool00000603oool010000000oooo0000
0000000C0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0CP3oool009L0oooo0`00
000H0?ooo`<000005@3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo04d0oooo002H
0?ooo`030000003oool0oooo01D0oooo0P00000H0?ooo`030000003oool0oooo00@0oooo00<00000
0?ooo`3oool0C03oool009P0oooo0`00000B0?ooo`<000006`3oool00`000000oooo0?ooo`030?oo
o`030000003oool0oooo04`0oooo002Z0?ooo`<000007P3oool00`000000oooo0?ooo`040?ooo`03
0000003oool0oooo04/0oooo002X0?ooo`8000008@3oool00`000000oooo0?ooo`030?ooo`800000
C@3oool00:D0oooo0`00000S0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0BP3o
ool00:<0oooo0P00000V0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0B@3oool0
0:00oooo0`00000X0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0B@3oool009h0
oooo0P00000[0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0B03oool009/0oooo
0`0000000`3oool000000?ooo`0Z0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0
A`3oool009P0oooo0`00000`0?ooo`030000003oool0oooo00L0oooo0P0000070?ooo`@00000?P3o
ool009H0oooo0P00000c0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0103oool0
0`000000oooo0?ooo`0o0?ooo`00T`3oool3000003D0oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`040?ooo`030000003oool0oooo03h0oooo002A0?ooo`80000000<0oooo0000003o
ool0=@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool0?@3oool006l0oooo0`00000L0?ooo`<00000>P3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo0080oooo00@000000?ooo`3oool00000?P3oool00700oooo00<000000?ooo`3oool0
6@3oool2000003d0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`020?ooo`800000
?`3oool006d0oooo1@00000G0?ooo`<00000?`3oool00`000000oooo0?ooo`0;0?ooo`800000A@3o
ool006d0oooo00@000000?ooo`3oool000005@3oool400000440oooo00<000000?ooo`3oool03@3o
ool00`000000oooo0?ooo`120?ooo`00KP3oool00`000000oooo0000000C0?ooo`800000A@3oool0
0`000000oooo0?ooo`0>0?ooo`030000003oool0oooo0440oooo001_0?ooo`800000403oool30000
04L0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`110?ooo`00OP3oool3000004X0
oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`100?ooo`00O03oool4000004/0oooo
00<000000?ooo`3oool0403oool00`000000oooo0?ooo`0o0?ooo`00NP3oool200000500oooo00<0
00000?ooo`3oool03P3oool200000440oooo001g0?ooo`<00000DP3oool00`000000oooo0?ooo`0@
0?ooo`030000003oool0oooo03h0oooo001d0?ooo`<00000E@3oool200000140oooo00<000000?oo
o`3oool0?P3oool00780oooo0P0000000`3oool000000?ooo`1D0?ooo`80000000<0]?Jd00000000
0000403oool00`000000oooo0?ooo`0m0?ooo`00K`3oool3000005P0oooo00H000000;Cf]0000000
]?Jd0:cb]02dm[@3000000h0oooo00<000000?ooo`3oool0?03oool006d0oooo0P00001J0?ooo`04
0000002dm[@0]?Jd000000H0]?Jd0P00000<0?ooo`800000?@3oool006X0oooo0`00001K0?ooo`0=
0000002dm[@0[?:d0;Cf]0000000]?Jd0:cb]02dm[@0[?:d0;Cf]02/l[@0]?Jd0:cb]002000000X0
oooo0P00000m0?ooo`00AP3oool4000001h0oooo0P0000000`3oool000000?ooo`1J0?ooo`030000
002dm[@0]?Jd0080]?Jd00<000000;Cf]02dm[@01@2dm[@2000000030;BZe@000000oooo00X0oooo
00<000000?ooo`3oool0>P3oool004H0oooo00<000000?ooo`3oool0703oool3000005h0oooo00`0
00000;Cf]02/l[@0]?Jd0:cb]02dm[@000000;Cf]02/l[@0]?Jd0:cb]02dm[@2000000040:bUc@2d
Z/d0[:G=000000/0oooo00<000000?ooo`3oool0>P3oool004L0oooo00<000000?ooo`3oool0603o
ool300000600oooo00<000000;Cf]02dm[@0102dm[@010000000]?Jd0;Cf]02dm[@2000000070;BZ
e@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0000000;0?ooo`030000003oool0oooo03T0oooo00180?ooo`03
0000003oool0oooo01D0oooo0P00001R0?ooo`0:0000002dm[@0[?:d0;Cf]02/l[@0]?Jd0:cb]02d
m[@000000;Cf]080000000T0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@000000303oool0
0`000000oooo0?ooo`0h0?ooo`00AP3oool010000000oooo0?ooo`00000C0?ooo`<0000000<0oooo
0000003oool0H03oool00`000000]?Jd0;Cf]0060;Cf]080000000`0]:[E0:bZc@2dZ]D0[:[=0;BZ
e@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@00000;0?ooo`800000>@3oool004L0oooo0P00000B0?oo
o`800000I@3oool02@000000]?Jd0:cb]02dm[@0[?:d0;Cf]02/l[@0]?Jd0:cb]0020000000=0;BZ
c@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0000000<0?ooo`030000
003oool0oooo00H0oooo0`00000^0?ooo`00F03oool3000006H0oooo00<000000;Cf]02dm[@01@2d
m[@20000000@0:bZc@000000[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZ
c@2dZ]D0[:[=000000`0oooo00<000000?ooo`3oool01P3oool00`000000oooo0?ooo`0]0?ooo`00
EP3oool2000006P0oooo00L000000;Cf]02/l[@0]?Jd0:cb]02dm[@0[?:d008000000180]:[=0:bU
c@2dZ/d000000;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZ
c@00000<0?ooo`030000003oool0oooo00<0oooo1@00000^0?ooo`00D`3oool3000006T0oooo00<0
00000;Cf]02dm[@00`2dm[@20000000E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d000000:bZc@2dZ]D0[:[=
0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0000000`0oooo00<00000
0?ooo`3oool00P3oool010000000oooo0?ooo`00000_0?ooo`00D03oool300000080oooo1`00001R
0?ooo`050000002dm[@0[?:d0;Cf]02/l[@00P0000005`2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2d
Z/d000000;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/
YLd0000000d0oooo00<000000?ooo`3oool00P3oool00`000000oooo0000000_0?ooo`00C@3oool3
000000<0oooo0`0000001P23KK@0NfVd08=]]01kJK@0Pffd07]Y]0P00000F@3oool010000000]?Jd
0;Cf]02dm[@20000000J09BAc@000000[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0000002/Z/d0
]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@00000;
0?ooo`8000001@3oool2000002l0oooo001;0?ooo`<00000103oool04@000000Y8Fd0:B1[0000000
NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/00800000E03o
ool40000000P0;Cf]02/l[@000000000002LTLd0U97=09bAc@2DTLd000000:bUc@2dZ/d0[:G=0;BZ
c@2/YLd0]:[=0000002dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bU
c@2dZ/d0[:G=0;BZc@00000=0?ooo`030000003oool0oooo03<0oooo00190?ooo`8000000`3oool0
5P000000oooo000000000000[8Fd0:B1[02/QK@0Y86/0000001kJK@0Pffd07]Y]023KK@0NfVd08=]
]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@2000004l0oooo1`0000008P2dm[@000000000002D
TLd0W97=09BAc@2LTLd0U97=09bAc@000000]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d000000:bZc@2d
Z]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@00
000=0?ooo`030000003oool0oooo0380oooo00160?ooo`<000001P3oool060000000Y86/0:B5]02T
PJ`0Y8Fd0:B1[02TQK@0000007]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01k
JK@0NfV/07]Y]000000083Rd000004/0oooo2`0000008@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2L
TLd0U97=09bAc@000000]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd000000:bUc@2dZ/d0[:G=0;BZc@2/
YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0000000=0?ooo`030000003o
ool0oooo0380oooo00140?ooo`800000203oool06P000000Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0
Y86/0000001kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0
0000020h]00P?;@00000B03oool;0000000T09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0
U97=09bAc@2DTLd000000:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0000002dZ]D0[:[=0;BZe@2/Z/d0
]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d000003@3oool00`00
0000oooo0?ooo`0a0?ooo`008P3oool2000001d0oooo100000070?ooo`80000001/0Y86/0:B5]02T
PJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]0000000NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01k
JK@0NfV/07]Y]01kJJ`0NfVd07]Y[000000063R/020h]0000000A@3oool;0000000V09bAc@2DTLd0
W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd000000:bUc@2dZ/d0
[:G=0;BZc@2/YLd000000:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0
]:[=0:bUc@2dZ/d0[:G=0;BZc@00000=0?ooo`800000<P3oool00240oooo00@000000?ooo`3oool0
00006@3oool3000000X0oooo01l000000:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd
0:B1[02/QK@0000008=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd
08=]]000000083bd020h]00P?;@0000004<0oooo2`000000:02LTLd0U97=09bAc@2DTLd0W97=09BA
c@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@000000]:[E0:bZc@2dZ]D0[:[=0;BZ
e@000000]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZ
c@2dZ]D0[:[=0;BZe@00000=0?ooo`030000003oool0oooo0300oooo000Q0?ooo`040000003oool0
oooo000001L0oooo0P00000;0?ooo`80000001l0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1
[02TQK@0Y86/0:B5]02TPJ`0000007]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y
]01kJJ`0NfVd07]Y[01kJK@00000020h]00H>:`083Rd00800000?`3oool<0000000Y09bAc@2DTLd0
W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0
00000;BZc@2/YLd0]:[=0:bUc@000000[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0
]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@0000003P3oool00`000000oooo0?ooo`0_0?oo
o`008@3oool010000000oooo0?ooo`00000D0?ooo`<00000303oool09@000000Y86/0:b5]02TPJ`0
[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]0000000Pffd07]Y]023KK@0
NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]000000083Rd020l]00P>;@0
83bd020h]0000000?03oool<0000000/09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=
09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@000000[:[=0;BZe@2/Z/d0]:[E
0000002dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=
0;BZe@2/Z/d0]:[E0:bZc@00000=0?ooo`030000003oool0oooo02l0oooo000Q0?ooo`040000003o
ool0oooo00000180oooo0P00000=0?ooo`80000002H0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/
0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0000007]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd
07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@00000020h]00H>:`083Rd01Ph[00P>;@063R/
000003T0oooo3@000000;P2/VKd0000009BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=
09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@000000[:G=0;BZc@2/YLd00000
0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=
0;BZc@2/YLd000000<g:e@00000=0?ooo`030000003oool0oooo02h0oooo000R0?ooo`800000403o
ool3000000h0oooo02X000000:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/
QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0000001kJK@0Pffd07]Y]023KK@0NfVd08=]]01k
JK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd0000000P?;@083Rd020l]00P>;@083bd020h]000
000g0?ooo`d0000002X0[9W50;BMa@000000W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2L
TLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=0000002dZ]D0[:[=0;BZe@00
0000]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2d
Z]D2000000030<g:e@3=c]h0000000h0oooo00<000000?ooo`3oool0;@3oool00340oooo0`00000@
0?ooo`80000002X0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0
Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0000007]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0
NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0000001Ph[00P>;@063R/020h]00H>:`083Rd01Ph[000000d
0?ooo`h000000340[9Vm0:bIa@2/VKd0000009BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0
U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd000000;BZc@2/YLd0
00000:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0
00000<g:gP3=b]D0cL[N0<g:e@3=b]h0000000`0oooo0P00000_0?ooo`00;`3oool4000000l0oooo
0P000000;@2dUKd000000:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0
Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0000001kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0
Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd0000000P?;@083Rd020l]00P>;@083bd020h]00P?;@0
83Rd0000000a0?ooo`l0000001/0[9W50;BMa@2/VLD0]9g50:bIa@000000U97=09bAc@2DTLd0W97=
09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=
0000002/Z/d00P0000005@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D0[:[=0;BZe@2/Z/d0]:[E
0:bZc@2dZ]D000000<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0000000>0?ooo`030000003oool0oooo
02`0oooo000/0?ooo`<000004P3oool0<@000000[9Fd0;BE_@2/UK@0]9Fm0000002TQK@0Y86/0:B5
]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0000
001kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y
]000000083Rd01Ph[00P>;@063R/020h]00H>:`083Rd01Ph[0000000;`3oool>0000000J0:bIa@2/
VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@000000W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2L
TLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd30000000H0:bUc@2dZ/d000000;BZ
c@2/YLd0]:[=0:bUc@2dZ/d0[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d000000<g:gP3=b]D0cL[N0<g:
e@3=b]h0cL[E0<g:gP3=b]D000003P3oool00`000000oooo0?ooo`0[0?ooo`00:P3oool2000001@0
oooo03<000000;BE]02dUKd0]9Fd0;BE_@2dUK@000000:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1
[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd00000023KK@0NfVd08=]
]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd0000000P>;@083bd020h
]00P?;@083Rd020l]00P>;@083bd020h]0000000;03oool?0000000H0;BMa@2/VLD0]9g50:bIa@2d
WLD0[9W50;BMa@2/VLD0]9g50000002LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2D
TLd0W97=09BAc@2LTLd0U97=0`000000702DTLd0W97=0000002dZ]D0[:[=0000002/Z/d0]:[E0:bZ
c@2dZ]D0[:[=0;BZe@2/Z/d0]:[E0:bZc@2dZ]D000000<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>
gP3=b]D0cLkN0<g:e@00000=0?ooo`030000003oool0oooo02/0oooo000X0?ooo`8000005@3oool0
=@000000[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd000000:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5
]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0000001kJJ`0NfVd07]Y
[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd0000000P>;@063R/020h
]00H>:`083Rd01Ph[00P>;@063R/020h]00H>:`0000002T0oooo0P0000000`1B000000000000000;
0000000G0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@000000U97=09bAc@2D
TLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd00P0000004@2DTLd0W97=09BAc@2L
TLd0U97=09bAc@000000]:[=0:bUc@000000[:G=0;BZc@2/YLd0]:[=0:bUc@2dZ/d0[:G=00800000
00`0cL[N0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP00000>0?ooo`030000
003oool0oooo02X0oooo000X0?ooo`800000503oool0=`000000]9Fd0;BE_@2dUK@0]9Fm0;BE]02d
UKd0]9Fd0000002TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02T
PJ`0[8Fd0:B1[02/QK@0Y86/0:b5]0000000Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023
KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@00000020l]00P>;@083bd020h]00P?;@083Rd020l]00P
>;@083bd020h]00000009P3oool20000000304X0001B0000000000`0000001D0]9g50:bIa@2dWLD0
[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@000000W97=09BAc@2LTLd0U97=09bAc@2DTLd0
W97=09BAc@2LTLd00`0000008P2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=0000002/Z/d0
]:[E0:bZc@000000[:[=0;BZe@2/Z/d0]:[E0:bZc@000000cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0
cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP00000=0?ooo`8000001P3oool300000280oooo
000X0?ooo`<000004@3oool20000000h0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd
0000002TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd
0:B1[02TQK@0Y86/0000001kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/
07]Y]01kJJ`0NfVd07]Y[000000063R/020h]00H>:`083Rd01Ph[00P>;@063R/020h]00H>:`083Rd
01Ph[000000S0?ooo`80000000<0DP0004X0001B00003@0000004`2/VLD0[9Vm0:bIa@2/VKd0[9W5
0:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0000009bAc@2DTLd0W97=09BAc@2LTLd00`00
00009P2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@000000[:G=
0;BZc@2/YLd000000:bUc@2dZ/d0[:G=0000003=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N
0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h000003@3oool00`000000oooo0?ooo`040?oo
o`040000003oool0oooo00000240oooo000X0?ooo`040000003oool0oooo000000l0oooo03`00000
0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd000000:b5]02TPJ`0[8Fd
0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/
0000001kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd
07]Y]000000083Rd020l]00P>;@083bd020h]00P?;@083Rd020l]00P>;@083bd020h]000000P0?oo
o`80000000@0BP000580001:0000DP00300000004P2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0
]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@000000U97=09bAc@<0000002T0W97=09BAc@2L
TLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd000000;BZe@2/
Z/d0]:[E0000002dZ]D0[:[=0000003=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0000000>0?ooo`030000003oool0oooo00<0oooo
00@000000?ooo`3oool000008@3oool002L0oooo00<000000?ooo`3oool00P3oool00`000000oooo
0?ooo`0;0?ooo`0n0000002dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm
0:bE]0000000Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/
0:B5]02TPJ`0Y8Fd0:B1[02TQK@0000007]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd
07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd0000000P>;@063R/020h]00H>:`083Rd01Ph[00P>;@063R/
020h]00H>:`083Rd01Ph[000000M0?ooo`80000000H0DP0004X0001B0000BP000580001:000;0000
000@0:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0
[9Vm0:bIa@<0000001L0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2D
TLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=0000002dZ/d0[:G=0;BZc@020000000D0<g:e@3=
b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP3=
b]D0cL[N0<g:e@00000>0?ooo`030000003oool0oooo0080oooo0`00000R0?ooo`009`3oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo00T0oooo040000000;BE_@2dUK@0]9Fm0;BE]02d
UKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@000000[8Fd0:B1[02/QK@0Y86/0:b5]02T
PJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0000007]Y]023
KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd0000000P
>;@083bd020h]00P?;@083Rd020l]00P>;@083bd020h]00P?;@083Rd020l]00P>;@000006P3oool2
0000000704X0001B0000BP000580001:0000DP0004X0000;0000000>0:bIa@2dWLD0[9W50;BMa@2/
VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD30000000I0;BMa@000000W97=09BA
c@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bA
c@2DTLd0W97=09BAc@000000[:[=0;BZe@030000000D0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>
gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP00000>0?ooo`03
0000003oool0oooo00<0oooo00<000000?ooo`3oool08@3oool002L0oooo0P0000050?ooo`800000
1`3oool20000000o0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd
0;BE_@2/UK@0]9Fm0000002TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd
0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0000001kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/
07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0000001Ph[00P>;@063R/020h]00H>:`083Rd
01Ph[00P>;@063R/020h]00H>:`083Rd008000005`3oool2000000080580001:0000DP0004X0001B
0000BP000580001:000:0000000=0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI
_@2/VLD0[9Vm0:bIa@030000000J0:bIa@2/VKd0[9W50:bI_@2/VLD0000009bAc@2DTLd0W97=09BA
c@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bA
c@2DTLd20000000H0<g:gP3=b]D0cL[N0000003=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E
0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h000003@3oool2000000@0
oooo0`00000Q0?ooo`009`3oool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00<0oooo
04<000000;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02d
UKd0]9Fd0;BE_@2dUK@000000:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02T
PJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0000007]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023
KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]000000083bd020h]00P?;@083Rd020l]00P
>;@083bd020h]00P?;@083Rd020l]00P>;@083bd00<000004`3oool20000000904X0001B0000BP00
0580001:0000DP0004X0001B0000BP0000X0000000`0]:G=0000002dWLD0[9W50;BMa@2/VLD0]9g5
0:bIa@2dWLD0[9W50;BMa@2/VLD20000000L0;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0
000009BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0
U97=09bAc@2DTLd0W97=09BAc@80000001X0cLkN0<g:e@3=c]h0cL[E0000003=b]D0cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=
b]D0cLkN000000h0oooo00<000000?ooo`3oool09@3oool002L0oooo00<000000?ooo`3oool01`3o
ool0C0000000oooo0?ooo`3oool000000:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd
0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0000002TQK@0Y86/0:B5]02TPJ`0Y8Fd
0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[0000000NfV/
07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`00000
01Ph[00P>;@063R/020h]00H>:`083Rd01Ph[00P>;@063R/020h]00H>:`083Rd01Ph[00P>;@00000
000@U000000?0?ooo`@0000000X0BP000580001:0000DP0004X0001B0000BP000580001:0000DP00
200000002`2/XLd0[:G=0:bQc@000000[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm00<0000001`0
[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50000002LTLd0U97=09bAc@2DTLd0
W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=0P000000702/
`^H0]<;V0000003=b]h0cL[E0<g:gP3=b]D000000<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=
b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@00000>0?ooo`030000
003oool0oooo02D0oooo000W0?ooo`030000003oool0oooo00P0oooo04`000000?ooo`0000000000
0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm
0;BE]02dUKd0]9Fd0000002TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/
0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]0000000Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd
07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd0000000P>;@083bd020h]00P?;@083Rd
020l]00P>;@083bd020h]00P?;@083Rd020l]00P>;@083bd0000000049@000002`3oool50000000;
0580001:0000DP0004X0001B0000BP000580001:0000DP0004X0001B00001`0000002P2dYLd0[:7=
0;BUc@2/XLd0]:G=0000002dWLD0[9W50;BMa@2/VLD30000000L0:bIa@2dWLD0[9W50;BMa@2/VLD0
]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0000009bAc@2DTLd0W97=09BAc@2LTLd0
U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@<0000001l0]<K^0;C2iP2da^h0]<;V0;C6kP00
0000cLkN0<g:e@3=c]h0cL[E0000003=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0000000h0oooo00<000000?ooo`3o
ool0903oool002H0oooo00<000000?ooo`0000002P3oool0C0000000_JFm0<FU_@000000]9Fm0:bE
]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE
_@2/UK@000000:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1
[02TQK@0Y86/0:B5]02TPJ`0000007]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y
]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]000000083Rd01Ph[00P>;@063R/020h]00H>:`083Rd01Ph
[00P>;@063R/020h]00H>:`083Rd01Ph[00P>;@00000000@U00000080?ooo`80000000l0012D0000
00000000BP000580001:0000DP0004X0001B0000BP000580001:0000DP0004X0001B00001`000000
1`2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0100000007@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bI
a@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0000002DTLd0W97=09BA
c@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=0080000001l0]<;V0:c2iP2d`^H0[<;V0;C2
iP2/`^H0]<;V0:c2iP000000cL[E0<g:gP3=b]D0cL[N0000003=b]h0cL[E0<g:gP3=b]D0cL[N0<g:
e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0080000000<0aMkV0000
003oool03@3oool00`000000oooo0?ooo`0S0?ooo`009P3oool2000000X0oooo04h000000;fU_@35
YLD0_JFm0000002dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2d
UK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@000000[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/
QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]0000000Pffd07]Y]023KK@0NfVd08=]]01k
JK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd0000000P>;@083bd020h]00P
?;@083Rd020l]00P>;@083bd020h]00P?;@083Rd020l]00P>;@083bd020h]00P?;@00000000@U000
00040?ooo`<000000140012D000000000000DP0004X0001B0000BP000580001:0000DP0004X0001B
0000BP000580001:0000DP0004X00005000000070:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@03
0000000M0:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2d
WLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50000002DTLd0W97=09BAc@2LTLd0U97=09bAc@2D
TLd00`0000009@2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0000003=
b]D0cLkN0<g:e@3=c]h000000<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D0cLkN0000003=h^h0aMkV0<gRkP0000003@3oool2000002D0oooo00080?oo
o`800000103oool2000000@0oooo0P0000040?ooo`8000000`3oool4000000<0oooo00<000000?oo
o`3oool0203oool0C`000000_JFm0<FU_@2mYKd0aJFm0000002dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE
]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]0000000Y86/0:B5
]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1
[02TQK@0000007]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y
]01kJJ`0NfVd07]Y[000000063R/020h]00H>:`083Rd01Ph[00P>;@063R/020h]00H>:`083Rd01Ph
[00P>;@063R/020h]00H>:`083Rd0000000049@0100000004`0039@0012D000000000000BP000580
001:0000DP0004X0001B0000BP000580001:0000DP0004X0001B0000BP000580001:00001@000000
1@2/XLd0[:G=0:bQc@2/YLd0[:7=00<0000001d0[:7=0:bUc@000000[9W50:bI_@2/VLD0[9Vm0:bI
a@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI
_@2/VLD0000009bAc@2DTLd0W97=09BAc@030000000Q0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2
iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0000003=b]D0cL[N0<g:e@3=b]h000000<g:
gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0080000000H0cMkV0<GN
iP3=g^H0aMkV0<gNiP00000>0?ooo`030000003oool0oooo0280oooo00070?ooo`040000003oool0
oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`03
0000003oool0oooo00@0oooo00<000000?ooo`3oool01`3oool0CP000000_JFm0<FUa@2mYKd0aJG5
0;fU_@35YLD000000;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd
0;BE_@2dUK@0]9Fm0;BE]02dUKd000000:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd
0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0000001kJK@0Pffd07]Y]023KK@0NfVd
08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@00000020l]00P>;@083bd
020h]00P?;@083Rd020l]00P>;@083bd020h]00P?;@083Rd020l]00P>;@083bd020h]0@000000P00
49@20000000@0580001:0000DP0004X0001B0000BP000580001:0000DP0004X0001B0000BP000580
001:0000DP0004X000@0000000@0]:G=0:bQc@2dYLd0[:7=0`0000007P2/XLd0]:G=0:bQc@2dYLd0
[:7=0;BUc@000000]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0
]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0000009bAc@80000002`0]<;V0;C6kP2d
`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2d
a^h0]<;V0000003=b]D0cLkN0<g:e@3=c]h000000<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=
b]D0cLkN0<g:e@000000aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV000000h0oooo00<000000?oo
o`3oool08P3oool000L0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo000000<0oooo00<000000?ooo`3oool00`3oool2000000L0oooo04`0
00000;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@000000[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0
[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd000000:B5]02TPJ`0
Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0
000007]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0
NfVd07]Y[01kJK@00000020h]00H>:`083Rd01Ph[00P>;@063R/020h]00H>:`083Rd01Ph[00P>;@0
63R/0`0000001023h_H0Rn;o0000002T_^h50000000>0580001:0000DP0004X0001B0000BP000580
001:0000DP0004X0001B0000BP000580001:0003000000040:bQc@2/YLd0[:7=0:bUc@80000001l0
[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd000000:bI_@2/VLD0[9Vm0:bIa@2/VKd0
[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0
[9Vm0080000002<0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0
]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H000000<g:gP3=b]D0cL[N0<g:e@000000
cL[E0<g:gP3=b]D0cL[N0<g:e@3=b]h0cL[E0<g:gP020000000:0<GNiP3=g^H0aMkV0<gNiP35g^H0
cMkV0<GNiP3=g^H0aMkV000000h0oooo00<000000?ooo`3oool08@3oool000030?ooo`0000000000
008000000P3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo00D0oooo
04X000000;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD000000;BE_@2dUK@0]9Fm0;BE]02d
UKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0000002T
PJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/
QK@0Y86/0000001kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01k
JK@0Pffd07]Y]023KK@0NfVd0000000P>;@083bd020h]00P?;@083Rd020l]00P>;@083bd020h]0<0
00001P2;h_l00`000000Y;k^0:BnkP030:BnkPT0000000D0DP0004X0001B0000BP00058000020000
00030:bQc@2dYLd000000080000001l0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0
[:7=0;BUc@2/XLd000000:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0
]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50080000002<0aLkN0000002d`^H0]<K^0;C2iP2da^h0
]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0
]<K^0;C2iP2da^h000000<g>gP3=b]D0cLkN0<g:e@000000cL[E0<g>gP3=b]D0cLkN0<g:e@020000
000=0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0000000>0?oo
o`030000003oool0oooo0200oooo00070?ooo`040000003oool0oooo000000P0oooo00@000000?oo
o`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000<0oooo
00<000000?ooo`3oool00`3oool2000000160;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0
_JFm0<FU_@000000]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0
]9Fm0:bE]02dUKd0[9Fd0;BE_@000000Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0
Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0000007]Y[01kJK@0NfV/07]Y]01kJJ`0
NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]000000083Rd01Ph[00P>;@0
63R/020h]0@0000001P0Pn;f08_Ro`23h_H0Rn;o08?RmP2;h_l0Pn;f08_Ro`23h_H0Rn;o0000002T
_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h50000001J0:bU
c@00000000000000002/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQ
c@2/YLd0[:7=0:bUc@000000[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bI
a@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@000000aM;V0<G>gP35d^H0aLkN0000002/`^H0]<;V0:c2
iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2
iP2/`^H0]<;V0:c2iP2d`^H000000<g:gP3=b]D0cL[N0<g:e@000000cL[E0<g:gP3=b]D000000<GN
iP3=g^H0aMkV0<gNiP35g^H0cMkV0<GNiP3=g^H0aMkV0<gNiP35g^H0cMkV0<GNiP3=g^H000003P3o
ool200000240oooo00080?ooo`8000002P3oool2000000@0oooo0P0000040?ooo`8000000`3oool0
0`000000oooo0?ooo`030?ooo`1600000035YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FU
a@2mYKd0aJG50;fU_@000000]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE
]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]0000000Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5
]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0000008=]]01kJK@0Pffd07]Y
]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@00000020l
]0<000003`2;h_l00`000000Y;k^0:BnkP0=0:BnkP<0000004H0]:G=0:bQc@2dYLd0[:7=0;BUc@2/
XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd000000:bIa@2d
WLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@00
0000cM;V0<G>gP3=d^H0aLkN0<gBiP000000]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2d
a^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP00
0000cLkN0<g:e@3=c]h0cL[E0`0000004@35g^H0cN;^0<GNiP3=h^h0aMkV0<gRkP35g^H0cN;^0<GN
iP3=h^h0aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0000000h0oooo00<000000?ooo`3oool07`3oool0
02D0oooo00<000000?ooo`3oool00P3oool0A@000000aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm
0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@000000[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd
0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]0000000Y86/0:B5]02TPJ`0Y8Fd
0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[0000000NfV/
07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0NfVd
07]Y[0030000000N08?RmP2;h_l0Pn;f08_Ro`23h_H0Rn;o08?RmP2;h_l0Pn;f08_Ro`23h_H0Rn;o
08?RmP2;h_l0Pn;f08_Ro`23h_H0Rn;o0000002T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^
0:BnkP2L_^h0Y;k^0`000000902T_^h0W;k^0000002/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0
[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd000000:bI_@2/VLD0
[9Vm0:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W50:bI_@80000003L0aM;V0<G>gP35
d^H0aLkN0<GBiP35c]h0aM;V0000002d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/
`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H000000<g:e@3=
b]h0cL[E0<g:gP000000cMkV0<GNiP3=g^H0aMkV0<gNiP35g^H0cMkV0<GNiP3=g^H0aMkV0<gNiP35
g^H0cMkV0<GNiP3=g^H0aMkV0<gNiP35g^H0000000l0oooo00<000000?ooo`3oool07P3oool002D0
oooo0P0000020?ooo`1400000035YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0
aJG50;fU_@35YLD0_JFm0<FUa@000000]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0
]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@000000[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0
[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]0000000Pffd07]Y]023KK@0
NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@20000000309cbiP2T
m^h0W?;V00@000003`2;h_l00`000000Y;k^0:BnkP050:BnkP<000001P2T_^h0A0000000[:7=0;BU
c@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQ
c@2dYLd000000;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50:bIa@2dWLD0[9W50000
0035c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H000000;C6kP2d`^H0]<K^0;C2
iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6
kP2d`^H0]<K^0;C2iP000000cL[E0P0000005@3=h^h000000<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV
0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0000000>0?ooo`03
0000003oool0oooo01h0oooo000U0?ooo`160000003oool0oooo00000035YKd0_JFm0<FU_@2mYKd0
aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd000000:bE]02dUKd0
[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0
00000:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0
Y86/0:B5]02TPJ`0Y8Fd0000001kJK@0NfV/07]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`0
NfVd07]Y[080000000T0W?K^09cbiP2Lm^h0W?;V09cfkP2Ll^H0W?K^09cbiP2Lm^h0100000004023
h_H0Rn;o08?RmP2;h_l0Pn;f08_Ro`23h_H0Rn;o08?RmP2;h_l0Pn;f0000002L_^h0Y;k^09bnkP2T
_^h30000000W0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^0000002/YLd0[:7=0:bU
c@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQ
c@2/YLd000000:bIa@2/VKd0[9W50:bI_@2/VLD0[9Vm0:bIa@2/VKd0[9W5008000000280aLkN0<GB
iP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H000000;C2iP2/`^H0]<;V0:c2
iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2
iP2/`^H0]<;V0P0000006035g^H0cMkV0<GNiP3=g^H000000<gNiP35g^H0cMkV0<GNiP3=g^H0aMkV
0<gNiP35g^H0cMkV0<GNiP3=g^H0aMkV0<gNiP35g^H0cMkV0<GNiP3=g^H0aMkV000000h0oooo0P00
00070?ooo`8000005@3oool002D0oooo04@000000?ooo`000000aJG50;fU_@35YLD0_JFm0<FUa@2m
YKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd000000;BE]02d
UKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@00
0000[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/
QK@0Y86/0:b5]02TPJ`0000007]Y]023KK@0NfVd08=]]01kJK@0Pffd07]Y]023KK@0NfVd08=]]080
000000l0W?;V0:CfkP2Ll^H0Y?K^09cbiP2Tm^h0W?;V0:CfkP2Ll^H0Y?K^09cbiP2Tm^h0W?;V0:Cf
kP2Ll^H01000000808_Ro`040000002T_^h00000000000d0Y;k^03l000000;BUc@2/XLd0]:G=0:bQ
c@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0000
002/VLD0]9g50:bIa@2dWLD0[9W50;BMa@2/VLD0]9g50000003=d^H0aLkN0<gBiP35c]h0cM;V0<G>
gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0000002d`^H0]<K^0;C2iP2da^h0]<;V0;C6
kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H00P000000
6P2di_l0[>;f00000035g^H0cN;^0<GNiP000000aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0<gR
kP35g^H0cN;^0<GNiP3=h^h0aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h000003P3oool2000000H0oooo
00@000000?ooo`3oool00000503oool002D0oooo0P000000@035YKd0_JFm0<FU_@2mYKd0aJFm0;fU
_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd000000;BE
_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE
]02dUKd000000:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5
]02TPJ`0Y8Fd0:B1[02TQK@0000007]Y]01kJJ`0NfVd07]Y[01kJK@0NfV/07]Y]01kJJ`20000000>
09cfkP2Ll^H0W?K^09cbiP2Lm^h0W?;V09cfkP2Ll^H0W?K^09cbiP2Lm^h0W?;V09cfkP2Ll^H40000
0080W;KV10000000102;h_l0Pn;f08_Ro`23h_H30000000Y09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^
09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0000002/XLd0[:G=0:bQc@2/YLd0[:7=
0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@000000[9W5
0:bI_@2/VLD0[9Vm0:bIa@2/VKd00P0000008@35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN
0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h000000:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V
0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP020000000M0:cRmP2di_H0[>;f
0;CVmP2/h_H000000<GNiP3=g^H0aMkV00000035g^H0cMkV0<GNiP3=g^H0aMkV0<gNiP35g^H0cMkV
0<GNiP3=g^H0aMkV0<gNiP35g^H0cMkV0<GNiP3=g^H0aMkV0<gNiP0000003P3oool00`000000oooo
0?ooo`040?ooo`040000003oool0oooo000001@0oooo000T0?ooo`80000003l0aJG50;fU_@35YLD0
_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0
aJG50;fU_@000000]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0
]9Fd0;BE_@2dUK@0]9Fm0;BE]0000000Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0
[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[0000000NfVd08=]]01kJK@0Pffd07]Y]023KK@0
0P000000302Ll^H0Y?K^09cbiP2Tm^h0W?;V0:CfkP2Ll^H0Y?K^09cbiP2Tm^h0W?;V0:CfkP@00000
00X0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV00000011QNH0@HG^1000000308_Ro`030000002T
_^h0Y;k^00d0Y;k^03X000000:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/
XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0000002dWLD0[9W50;BMa@2/VLD000000<G>gP3=
d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35
c]h0cM;V0<G>gP000000]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0;C6kP2d`^H0]<K^0;C2iP2d
a^h0]<;V0P0000007`2di_l0[>;f0;CVo`2/h_H0]>Ko0:cRmP2di_l0[>;f00000035g^H0cN;^0<GN
iP000000aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0<gR
kP35g^H0cN;^0<GNiP0000003`3oool00`000000oooo0?ooo`040?ooo`8000005@3oool002@0oooo
0P000000?@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0
aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@000000[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0
[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0000002TQK@0Y86/0:B5]02TPJ`0
Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[0000000
NfV/07]Y]01kJJ`00P0000002P2Lm^h0W?;V09cfkP2Ll^H0W?K^09cbiP2Lm^h0W?;V09cfkP2Ll^H4
000000/0W;KV100000000`0iQNH0@HG^000000020000001908_Ro`23h_H0000009bnkP2T_^h0W;k^
0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h000000:bUc@2/XLd0[:G=
0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=
0000002/VKd0[9W50:bI_@000000aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V
0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h000000:c2iP2d`^H0[<;V
0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP020000000P0:cRmP2di_H0[>;f0;CVmP2/h_H0]>Kf
0:cRmP2di_H0[>;f0;CVmP2/h_H000000<GNiP3=g^H0aMkV00000035g^H0cMkV0<GNiP3=g^H0aMkV
0<gNiP35g^H0cMkV0<GNiP3=g^H0aMkV0<gNiP35g^H0cMkV0<GNiP3=g^H2000000l0oooo00<00000
0?ooo`3oool00P3oool010000000oooo0?ooo`00000D0?ooo`00903oool0?@000000oooo0000002m
YKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35
YLD0_JFm0<FUa@2mYKd0aJG50000002dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2d
UK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@000000:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/
QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd00000023KK@00P0000000`2L
l^H0Y?K^09cbiP090000000=09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L
]^H0Y;K^09bfiP04000000070465kP11QNH0@HG^0465iP11QNh0@HGV0465kP04000000h0Y;k^01<0
00000;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd0
]:G=0:bQc@2dYLd0[:7=00<0000001l0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0
cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP000000
]<;V0;C6kP2d`^H0]<K^0;C2iP2da^h0]<;V0080000001h0]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H0
]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H0]>Ko0:cRmP000000aMkV0<gRkP000000cN;^0<GNiP3=h^h0
aMkV0<gRkP35g^H0cN;^0<GNiP3=h^h0aMkV0<gRkP35g^H4000000030:CFo`2Td_H0000000h0oooo
0P0000040?ooo`8000005@3oool002@0oooo03/000000?ooo`000000aJFm0;fU_@35YKd0_JFm0<FU
_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU
_@000000[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE
_@2/UK@0]9Fm0:bE]0000000Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1
[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[007000000@0]?K^0P0000000`35a]D0000009bfiP0:09bf
iP<00000102L]^H02P000000@HG^03V5iP11QNh0>HGV0465kP0iQNH0@HG^03V5iP11QNh30000000M
0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0000002/XLd0[:G=
0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd0[:7=0:bUc@2/XLd0[:G=0:bQc@2/YLd00`00
00007`35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN
0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H000000;C2iP2/`^H0]<;V
0:c2iP2d`^H00P0000007@2/h_H0]>Kf0:cRmP2di_H0[>;f0;CVmP2/h_H0]>Kf0:cRmP2di_H0[>;f
0;CVmP2/h_H0]>Kf0:cRmP2di_H0[>;f00000035g^H0cMkV0000003=g^H0aMkV0<gNiP35g^H0cMkV
0<GNiP3=g^H0aMkV00<0000000L0W=;f0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0000000d0oooo0`00000K
0?ooo`00903oool20000000g0?ooo`000000aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35
YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@000000]9Fd0;BE_@2d
UK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0000002/
QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5]002
0000000;0;CfkP2mn_H0]?K^0;gjmP2dm^h0_O[f0;CfkP2mn_H0]?K^0;gjmP0000001035a]h02@00
0000W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP020000000C09bfiP2T]^h0W;KV0:BfkP2L
]^H0Y;K^09bfiP2T]^h000000465kP11QNH0@HG^0465iP11QNh0@HGV0465kP11QNH0@HG^0465iP05
000000T0Y;k^00`000000:bQc@2dYLd0[:7=0;BUc@2/XLd0]:G=0:bQc@2dYLd0[:7=0;BUc@2/XLd4
000000<0W<Kf01d000000<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0
aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP000000
]<K^0;C2iP020000000L0;CVo`2/h_H0]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H0]>Ko0:cRmP2di_l0
[>;f0;CVo`2/h_H0]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H000000<GNiP3=h^h000000<gRkP35g^H0
cN;^0<GNiP<0000000/0Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H0Y=Ko0000000?
0?ooo`030000003oool0oooo01P0oooo000T0?ooo`0i0000003oool0oooo0000002mYKd0aJFm0;fU
_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU
_@2mYKd0aJFm0000002dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE
]02dUKd0[9Fd0;BE_@2/UK@000000:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd0:B1[02TQK@0Y86/0:B5
]02TPJ`0Y8Fd0:B1[02TQK@0000000/0]?K^00P000000<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0000
00@0W;KV0`00000:09bfiP0@0000000iQNH0@HG^03V5iP11QNh0>HGV0465kP0iQNH0@HG^03V5iP11
QNh0>HGV0000000aQNH0<HG^0365iP@0000000d0Y;k^09bnkP2T_^h0W;k^0:BnkP000000[:G=0:bQ
c@2/YLd0[:7=0:bUc@2/XLd0[:G=00@000000280U<;f09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f09c6
mP000000aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>
gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0P0000006@2/h_H0]>Kf
0:cRmP2di_H0[>;f0;CVmP2/h_H0]>Kf0:cRmP2di_H0[>;f0;CVmP2/h_H0]>Kf0:cRmP2di_H0[>;f
0;CVmP2/h_H0]>Kf0:cRmP2di_H0[>;f00000035g^H01@0000003`2Ld_H0Y=Ko09cBmP2Te_l0W=;f
0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0W=;f0:CFo`2Ld_H0Y=Ko0000000>0?ooo`030000003oool0oooo
01P0oooo000M0?ooo`800000103oool00`000000oooo0?ooo`020?ooo`0b0000002mYKd0aJG50;fU
_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FU
a@2mYKd000000;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE
]02dUKd0]9Fd0;BE_@000000[8Fd0:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@0Y86/0:b5
]02TPJ`20000000:0;gjmP2dm^h0_O[f0;CfkP2mn_H0]?K^0;gjmP2dm^h0_O[f0;CfkP8000002035
a]h40000000Q0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:Bf
kP2L]^H000000465iP11QNh0@HGV0465kP11QNH0@HG^0465iP11QNh0@HGV0465kP11QNH000000365
kP0iRNh0<HG^03V9kP0aQNh0>HW^00@000000P2T_^h00`000000]:G=0:bQc@04000000d0W<Kf01L0
00000<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0
cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP020000000I0;CjmP00000000000:cRmP2di_l0
[>;f0;CVo`2/h_H0]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H0]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H0
]>Ko0:cRmP2di_l0[>;f0;CVo`040000000B0:CBmP2Te_l0Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0
Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H000003`3oool00`000000oooo0?ooo`0G
0?ooo`00703oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo0080oooo03000000
0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm
0;fU_@35YKd0_JFm0<FU_@2mYKd000000:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd
0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@000000Y8Fd0:B1[02TQK@0Y86/0:B5]02TPJ`0Y8Fd
0:B1[02TQK@2000000/0]?K^00T000000<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h01000
000A09bfiP0F0000000iQNH0@HG^03V5iP11QNh0>HGV0465kP0iQNH0@HG^03V5iP11QNh0>HGV0000
000aQNH0<HG^0365iP0aQNh0<HGV0365kP0aQNH0<HG^0365iPD0000002H0U<;f09c6mP2D`_H0W<Kf
09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP2La_H00000
0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V
0<G>gP35d^H0aLkN0<GBiP80000001L0[?K^0;CfkP2/m^h0]?K^0:cfkP000000[>;f0;CVmP2/h_H0
]>Kf0:cRmP2di_H0[>;f0;CVmP2/h_H0]>Kf0:cRmP2di_H0[>;f0;CVmP2/h_H0]>Kf0:cRmP040000
0080W=ko01D0000009cBmP000000W=;f0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0W=;f0:CFo`2Ld_H0Y=Ko
09cBmP2Te_l0W=;f0:CFo`2Ld_H0Y=Ko09cBmP0000003P3oool2000001P0oooo000L0?ooo`040000
003oool0oooo000000<0oooo00<000000?ooo`3oool00`3oool0?0000000aJG50;fU_@35YLD0_JFm
0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG5
0000002dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd
0;BE_@2dUK@000000:B1[02/QK@0Y86/0:b5]02TPJ`0[8Fd0:B1[02/QK@000000;gjmP2dm^h0_O[f
0;CfkP2mn_H0]?K^0;gjmP2dm^h0_O[f0;CfkP2mn_H0]?K^000000H0aLKN0`0000040<G6gP0[0000
002L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bf
iP2T]^h0W;KV00000011QNH0@HG^0465iP11QNh0@HGV0465kP11QNH0@HG^0465iP11QNh0@HGV0000
000aQNh0>HW^0365kP0iRNh0<HG^03V9kP0aQNh0>HW^0365kP0iRNh0<HG^0000000D09c6mP0K0000
003=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gB
iP35c]h0cM;V0000002dn_H0]?K^0;CjmP2dm^h0]?[f0;CfkP2dn_H0]?K^0080000000`0]>Ko0:cR
mP2di_l0[>;f0;CVo`2/h_H0]>Ko0:cRmP2di_l0[>;f0;CVo`2/h_H3000000L0W=ko01@000000:CB
mP000000Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CF
o`2Td_H0Y=Ko000000l0oooo00<000000?ooo`3oool05P3oool001`0oooo00@000000?ooo`3oool0
00000`3oool2000000@0oooo02`000000;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm
0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd000000;BE_@2/UK@0]9Fm
0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0000002TQK@0Y86/
0:B5]02TPJ`0Y8Fd0:B1[08000002`2dm^h2000000040<G6e@35a]h0aLKE0<G6gP<0000000P0aLKN
0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN00000180W;KV03`0000003V5iP11QNh0>HGV0465kP0iQNH0
@HG^03V5iP11QNh0>HGV0465kP000000<HG^0365iP0aQNh0<HGV0365kP0aQNH0<HG^0365iP0aQNh0
<HGV0365kP0aQNH0000009C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0
W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP000000aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0
aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h20000000D0;CfkP2/m^h0]?K^0:cfkP2d
m^h0[?K^0;CfkP2/m^h0]?K^0:cfkP2dm^h000000;CVmP2/h_H0]>Kf0:cRmP2di_H0[>;f0;CVmP2/
h_H3000000/0W=ko0P0000004P2Te_l0W=;f0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0W=;f0:CFo`2Ld_H0
Y=Ko09cBmP2Te_l0W=;f0:CFo`2Ld_H0Y=Ko000000l0oooo00<000000?ooo`3oool05@3oool001`0
oooo00@000000?ooo`3oool000000`3oool00`000000oooo0?ooo`040?ooo`0h0000002mYKd0aJG5
0;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm
0<FUa@2mYKd000000;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm
0;BE]02dUKd0]9Fd0000002TPJ`0[8Fd0:B1[02/QK@0Y86/0000002dm^h0_O[f0;CfkP2mn_H0]?K^
0;gjmP2dm^h0_O[f0;CfkP2mn_H0]?K^0;gjmP0000030<G6gP<000002`35a]h0;0000000Y;K^09bf
iP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:Bf
kP000000@HG^0465iP11QNh0@HGV0465kP11QNH0@HG^0465iP11QNh0@HGV0465kP000000>HW^0365
kP0iRNh0<HG^03V9kP0aQNh0>HW^0365kP0iRNh0<HG^03V9kP0aQNh000004`2La_H03@000000aLkN
0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H00P0000003P2dm^h0]?[f
0;CfkP2dn_H0]?K^0;CjmP2dm^h0]?[f0;CfkP2dn_H0]?K^0;CjmP2dm^h0]?[f0P0000000`2/h_H0
]>Ko00000003000000l0W=ko0P0000004P2Te_l0Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CF
o`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H0Y=Ko000000h0oooo00<000000?ooo`3oool05@3o
ool001d0oooo0P0000040?ooo`030000003oool0oooo00@0oooo02P000000<FU_@2mYKd0aJFm0;fU
_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU
_@000000]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE
]02dUKd0[9Fd0000002TPJ`0Y8Fd0P00000<0;CfkP0300000035a]h0aLKE008000000100aLKN0<G6
e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h000004@2L
]^H0=`000000@HG^03V5iP11QNh0>HGV0465kP0iQNH0@HG^03V5iP11QNh0>HGV0465kP000000<HG^
0365iP0aQNh0<HGV0365kP0aQNH0<HG^0365iP0aQNh0<HGV0365kP0aQNH0000009C2mP2La_H0U<;f
09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf
09C2mP000000aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h00P0000004@2dm^h0[?K^
0;CfkP2/m^h0]?K^0:cfkP2dm^h0[?K^0;CfkP2/m^h0]?K^0:cfkP2dm^h0[?K^0;CfkP2/m^h0]?K^
00<000005@2Lg_l04P000000Y=Ko09cBmP2Te_l0W=;f0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0W=;f0:CF
o`2Ld_H0Y=Ko09cBmP2Te_l0W=;f000000l0oooo0P00000E0?ooo`008`3oool00`000000oooo0?oo
o`050?ooo`0T00000035YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU
_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD000000;BE_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE
_@2dUK@0]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0`000000302mn_H0]?K^0;gjmP2dm^h0_O[f
0;CfkP2mn_H0]?K^0;gjmP2dm^h0_O[f0;CfkPD000004P35a]h040000000Y;K^09bfiP2T]^h0W;KV
0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h50000000G0465kP11QNH0
@HG^0465iP11QNh0@HGV0465kP11QNH0@HG^0000000iRNh0<HG^03V9kP0aQNh0>HW^0365kP0iRNh0
<HG^03V9kP0aQNh0>HW^0365kP0000004`2La_H06P000000aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0
aLkN0000002dm^h0]?[f0;CfkP2dn_H0]?K^0;CjmP2dm^h0]?[f0;CfkP2dn_H0]?K^0;CjmP2dm^h0
]?[f0;CfkP2dn_H0]?K^0P0000001023i_H0Nn;^00000000000E09cNo`0B0000002Te_l0Y=;f0:CF
o`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H000003`3o
ool00`000000oooo0?ooo`0C0?ooo`008P3oool2000000L0oooo028000000;fU_@35YKd0_JFm0<FU
_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU
_@000000[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd0;BE_@2/UK@0]9Fm0:bE]02dUKd0[9Fd4`00
00005`35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN
0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0000000T0W;KV1@0000001@2LVLd0W9G=09bI
c@000000LhGN00D0000002/0@HG^03V5iP11QNh0>HGV0465kP000000<HG^0365iP0aQNh0<HGV0365
kP0aQNH0<HG^0365iP0aQNh0<HGV0365kP0aQNH0000009C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2
mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP000000aLkN0<GB
iP35c]h0aM;V0080000000l0[?K^0;CfkP2/m^h0]?K^0:cfkP2dm^h0[?K^0;CfkP2/m^h0]?K^0:cf
kP2dm^h0[?K^0;CfkP2/m^h00`0000001`1kh^h0PnKf07_RkP23i_H0Nn;^08?VmP000000502Lg_l2
0000000A0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0W=;f0:CFo`2Ld_H0Y=Ko09cBmP2Te_l0W=;f0:CFo`2L
d_H0Y=Ko09cBmP0000003P3oool00`000000oooo0?ooo`0C0?ooo`008P3oool00`000000oooo0?oo
o`070?ooo`0L0000002mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FU
a@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@000000]9Fm0;BE]02dUKd0]9Fd0;BE_@2dUK@0]9Fm0;BE
]0D0000000d0LmKo07_Jo`1ce_l0Nm[o07?Fo`1kf_l0LmKo07_Jo`1ce_l0Nm[o07?Fo`1kf_l0LmKo
00<0000000<0]<;V00000035a]h05`35a]h010000000Y;K^09bfiP2T]^h50000000?09bIe@2LULd0
W9WE09bEc@2LVMD0W9G=09bIe@2LULd0000006Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN00D0000000l0
@HG^0000000iRNh0<HG^03V9kP0aQNh0>HW^0365kP0iRNh0<HG^03V9kP0aQNh0>HW^0365kP000000
4`2La_H04P000000aLkN000000000000]?[f0;CfkP2dn_H0]?K^0;CjmP2dm^h0]?[f0;CfkP2dn_H0
]?K^0;CjmP2dm^h0]?[f0;CfkP<0000000X0Nn;^08?VmP1kh^h0PnKf07_RkP23i_H0Nn;^08?VmP1k
h^h0PnKf0P00000B09cNo`/0000000P0Y=Ko0:CBmP2Te_l0Y=;f0:CFo`2Td_H0Y=Ko000000l0oooo
00<000000?ooo`3oool04P3oool00280oooo00<000000?ooo`3oool01`3oool05`000000aJFm0;fU
_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU
_@2mYKd0aJFm0000002dUKd0[9Fd00D0000000l0LmKo07_Fo`1ce_l0NmKo07?Fo`1ke_l0LmKo07_F
o`1ce_l0NmKo07?Fo`1ke_l0LmKo07_Fo`1ce_l00`0000007P2/`^H0]<;V0:c2iP2d`^H0[<;V0000
0035a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6
e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP<0000001T0W9W=09bEc@2LVLd0W9G=
09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@000000JXGE07>5gP1ZQMD0LhGN
06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN00D0000001`0<HG^0365iP0aQNh0<HGV0365kP0aQNH0<HG^
0365iP0aQNh0<HGV0000002D`_H0W<Kf09C2mP2La_H0U<;f09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f
09c6mP2D`_H0W<Kf09C2mP2La_H0U<;f1@000000302/m^h0]?K^0:cfkP2dm^h0[?K^0;CfkP2/m^h0
]?K^0:cfkP2dm^h0[?K^0;CfkP8000000100Nn;^08?VmP1kh^h0PnKf07_RkP23i_H0Nn;^08?VmP1k
h^h0PnKf07_RkP23i_H0Nn;^08?VmP1kh^h000002`2Lg_l60000000;09bjkP2D]^h000000000001k
Z_H0NjG^07^ZmP1kYNh0Nj[f07^UkP1kZ_H02@00000?0?ooo`030000003oool0oooo00D0oooo1@00
00020?ooo`8000000`3oool00280oooo00<000000?ooo`3oool0203oool04`000000aJG50;fU_@35
YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm0<FUa@2m
YKd00`0000004@1ce_l0Nm[o07?Fo`1kf_l0LmKo07_Jo`1ce_l0Nm[o07?Fo`1kf_l0LmKo07_Jo`1c
e_l0Nm[o07?Fo`1kf_l0LmKo00@0000000T0]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP00
00005@35a]h30000000P09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0
W9WE09bEc@2LVMD0W9G=09bIe@000000LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0
JXGE07>5gP1ZQMD0LhGN0000001BVOH50000000703V9kP0aQNh0>HW^0365kP0iRNh0<HG^0000000:
09c6mPH0000000D0Lm[f07_Jo`1cf_H0Nm[o07?JmP05000000040;CfkP2dn_H0]?K^0;CjmP<00000
0180PnKf07_RkP23i_H0Nn;^08?VmP1kh^h0PnKf07_RkP23i_H0Nn;^08?VmP1kh^h0PnKf07_RkP23
i_H0Nn;^08?VmP1kh^h2000000<0W=ko1P000000202L]^h0W;[^09bfkP2L^^h0W;K^09bjkP2L]^h0
W;[^0P0000004@1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^ZmP1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^ZmP1k
Z^h0Nj[f07^ZkP1kZ_H0000000h0oooo0P0000080?ooo`050000003oool0oooo0?ooo`0000000P3o
ool100000040oooo0@3oool00280oooo00<000000?ooo`3oool0203oool04@000000_JFm0<FU_@2m
YKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@2mYKd0aJFm00800000
00<0_M;V0;gFkP0000000`0000002`1ce_l0NmKo07?Fo`1ke_l0LmKo07_Fo`1ce_l0NmKo07?Fo`1k
e_l0LmKo00D0000001h0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP00
0000aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35
a]D0aLKN0<G6e@35a]h2000000<0]:k=02D0000009bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2L
VLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@000000LhGN06Z5e@1cQMh0JXGE07>5gP1Z
QMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN0000001BUOH0BYG^05:EmP1:UNh0DYGf00H00000
00D0<HGV0000002D`_H0W<Kf09C2mP060000000@07?Jo`1cf_H0Lm[o07?JmP1cf_l0Lm[f07?Jo`1c
f_H0Lm[o07?JmP1cf_l0Lm[f07?Jo`1cf_H0Lm[o07?JmPL0000000d0Nn;^08?VmP1kh^h0PnKf07_R
kP23i_H0Nn;^08?VmP1kh^h0PnKf07_RkP23i_H0Nn;^00X0000000l0W;[^09BfkP2L^^h0U;K^09bj
kP2D]^h0W;[^09BfkP2L^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h00P000000401kYNh0Nj[f07^U
kP1kZ_H0NjG^07^ZmP1kYNh0Nj[f07^UkP1kZ_H0NjG^07^ZmP1kYNh0Nj[f07^UkP00000?0?ooo`03
0000003oool0oooo00H0oooo00D000000?ooo`3oool0oooo000000020?ooo`4000000@3oool10?oo
o`008P3oool2000000X0oooo00h000000;fU_@35YLD0_JFm0<FUa@2mYKd0aJG50;fU_@35YLD0_JFm
0<FUa@2mYKd0aJG50;fU_@80000000P0_MK^0;gBkP2me^h0_M;^0;gFkP2md^h0_MK^0;gBkP<00000
00H0LmKo07_Jo`1ce_l0Nm[o07?Fo`1kf_l20000000B07?Fo`00000000000;C6iP2d`^H0]<KV0;C2
iP2da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H000004035a]h0<@000000_JkE
0;B^c@2m[]D0]:k=0;f^e@000000W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=
09bIe@2LULd0W9WE09bEc@2LVMD0W9G=0000001ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN
06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN0000001BVOH0BYG^05:ImP1:UNh0DYWf04ZEkP1BVOH0BYG^
05:ImP1:UNh01@000000501cf_H0Nm[o07?JmP1kf_l0Lm[f07_Jo`1cf_H0Nm[o07?JmP1kf_l0Lm[f
07_Jo`1cf_H0Nm[o07?JmP1kf_l0Lm[f07_Jo`1cf_H0Nm[o0`0000001P2Di_l0RnKo09CVo`2;i_l0
U>Ko08_Vo`d0000001X0PkKf08^jmP23]_H0Rk[f08>fmP2;^_H0PkKf08^jmP000000W;[^09bfkP2L
^^h0W;K^09bjkP2L]^h0W;[^09bfkP2L^^h0W;K^09bjkP2L]^h0W;[^09bfkP2L^^h0W;K^09bjkP80
00000100Nj[^07^ZmP1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^ZmP1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^Z
mP1kZ^h000003`3oool00`000000oooo0?ooo`050?ooo`050000003oool0oooo0?ooo`0000000P3o
ool100000040oooo0@3oool00240oooo00<000000?ooo`3oool02P3oool030000000aJFm0;fU_@35
YKd0_JFm0<FU_@2mYKd0aJFm0;fU_@35YKd0_JFm0<FU_@80000000d0_M;V0;gFkP2md^H0_MK^0;gB
iP2me^h0_M;V0;gFkP2md^H0_MK^0;gBiP2me^h0_M;V00H0000001@0NmKo000000000000[<;V0;C2
iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2
iP80000000`0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0<G6e@35a]h0aLKE0<G6gP35a]D20000
00L0]:k=02h0000009bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=
09bIc@2LULd0W9W=09bEc@000000JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN
06Z5e@1cQMh0JXGE07>5gP000000DYGf04ZEkP1BUOH0BYG^05:EmP1:UNh0DYGf04ZEkP1BUOH0BYG^
05:EmP1:UNh0DYGf0`0000003`1cf_l0Lm[f07?Jo`1cf_H0Lm[o07?JmP1cf_l0Lm[f07?Jo`1cf_H0
Lm[o07?JmP1cf_l0Lm[f07?Jo`030000000:09CVo`2;h_H0U>Ko08_RmP2Di_l0Rn;f09CVo`2;h_H0
U>Ko08_RmP80000000<0]<[^0:c:iP0000003`23]_H090000000U;K^09bjkP2D]^h0W;[^09BfkP2L
^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0000007^ZmP1k
YNh0Nj[f07^UkP1kZ_H0NjG^07^ZmP1kYNh0Nj[f07^UkP1kZ_H0NjG^07^ZmP1kYNh0Nj[f000000l0
oooo00<000000?ooo`3oool01@3oool01@000000oooo0?ooo`3oool000000080oooo0@0000010?oo
o`40oooo000Q0?ooo`030000003oool0oooo00/0oooo00T000000<FUa@2mYKd0aJG50;fU_@35YLD0
_JFm0<FUa@2mYKd00P0000003`2me^h0_M;^0;gFkP2md^h0_MK^0;gBkP2me^h0_M;^0;gFkP2md^h0
_MK^0;gBkP2me^h0_M;^0;gFkP030000000J0;gFkP00000000000000002da^H0]<;V0;C6iP2d`^H0
]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0
]<;V0;C6iP0000090<G6gP80000003X0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0
000009bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0
W9G=09bIe@000000LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0
LhGN06Z5e@1cQMh0000005:ImP1:UNh0DYWf04ZEkP1BVOH0BYG^05:ImP1:UNh0DYWf04ZEkP1BVOH0
BYG^05:ImP1:UNh0DYWf0P0000002@1cf_H0Nm[o07?JmP1kf_l0Lm[f07_Jo`1cf_H0Nm[o07?JmP04
0000000;09CVo`2;i_l0U>Ko08_Vo`2Di_l0RnKo09CVo`2;i_l0U>Ko08_Vo`2Di_l00P0000009`2d
b^h0]<k^0;C:kP2dc^h0]<[^00000023]_H0Rk[f08>fmP2;^_H0PkKf08^jmP23]_H0Rk[f08>fmP2;
^_H0PkKf08^jmP23]_H0Rk[f08>fmP000000W;K^09bjkP2L]^h0W;[^09bfkP2L^^h0W;K^09bjkP2L
]^h0W;[^09bfkP2L^^h0W;K^09bjkP2L]^h0W;[^09bfkP020000000@07^ZmP1kZ^h0Nj[f07^ZkP1k
Z_H0Nj[^07^ZmP1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^ZmP1kZ^h0Nj[f000000h0oooo0P0000050?oo
o`800000103oool2000000<0oooo000Q0?ooo`030000003oool0oooo00/0oooo00L000000;fU_@35
YKd0_JFm0<FU_@2mYKd0aJFm0080000000h0_M;V0;gFkP2md^H0_MK^0;gBiP2me^h0_M;V0;gFkP2m
d^H0_MK^0;gBiP2me^h0_M;V0;gFkP<0000002H0_MK^0;gBiP2me^h0_M;V0000003=b]D000000:c2
iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2
iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP000000aLKN0<G6e@35a]h0aLKE0<G6gP35a]D0aLKN0000
00`0]:k=0380000009bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=
09bEc@2LVLd0W9G=09bIc@000000LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE
07>5gP1ZQMD0LhGN06Z5e@000000BYG^05:EmP1:UNh0DYGf04ZEkP1BUOH0BYG^05:EmP1:UNh0DYGf
04ZEkP1BUOH0BYG^05:EmP1:UNh0DYGf04ZEkP<0000000<0Lm[f07?Jo`1cf_H00`0000003@2;h_H0
U>Ko08_RmP2Di_l0Rn;f09CVo`2;h_H0U>Ko08_RmP2Di_l0Rn;f09CVo`2;h_H00P0000002@2db^h0
[<[V0;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V0000000?08>fmP0B0000002D]^h0W;[^09BfkP2L^^h0
U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0U;K^0P0000003`1k
Z_H0NjG^07^ZmP1kYNh0Nj[f07^UkP1kZ_H0NjG^07^ZmP1kYNh0Nj[f07^UkP1kZ_H0NjG^0000000;
0?ooo`<0000000<0oooo0000003oool03`3oool00240oooo00<000000?ooo`000000303oool01000
0000_JFm0<FUa@2mYKd20000000=0;gFkP2md^h0_MK^0;gBkP2me^h0_M;^0;gFkP2md^h0_MK^0;gB
kP2me^h0_M;^0;gFkP03000000050;gFkP2md^h0_MK^0;gBkP2me^h00P0000006`3=b]D0cLkN0<g:
e@000000]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0;C6
iP2d`^H0]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H0]<KV000000040<G6gP8000000480_JkE0;B^c@2m
[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0000002LVMD0W9G=09bIe@2L
ULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0000006Z5e@1c
QMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE0000001:
UNh0DYWf04ZEkP1BVOH0BYG^05:ImP1:UNh0DYWf04ZEkP1BVOH0BYG^05:ImP1:UNh0DYWf04ZEkP1B
VOH0BYG^05:ImP1:UNh30000000>08_Vo`2Di_l0RnKo09CVo`2;i_l0U>Ko08_Vo`2Di_l0RnKo09CV
o`2;i_l0U>Ko08_Vo`2Di_l20000000n0;C:kP2dc^h0]<[^0;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^
0;C>kP2db^h0000008>fmP2;^_H0PkKf08^jmP23]_H0Rk[f08>fmP2;^_H0PkKf08^jmP23]_H0Rk[f
08>fmP2;^_H0PkKf0000002L]^h0W;[^09bfkP2L^^h0W;K^09bjkP2L]^h0W;[^09bfkP2L^^h0W;K^
09bjkP2L]^h0W;[^09bfkP2L^^h0W;K^0000001kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^ZmP1kZ^h0Nj[f
07^ZkP1kZ_H0Nj[^07^ZmP1kZ^h0Nj[f07^ZkP0000060?ooo`@00000103oool00`000000oooo0?oo
o`0>0?ooo`008@3oool2000000d0oooo010000000<FU_@00000000000;gBiP2me^h0_M;V0;gFkP2m
d^H0_MK^0;gBiP2me^h0_M;V0;gFkP2md^H0_MK^0`0000009`2me^h0_M;V0;gFkP2md^H0_MK^0;gB
iP2me^h000000<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0000002/`^H0]<;V0:c2iP2d`^H0[<;V0;C2
iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2
iP2d`^H000000<G6gP0200000100]:k=0340000009bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2L
ULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@000000JXGE07>5gP1ZQMD0LhGN06Z5e@1c
QMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0000004ZEkP1BUOH0BYG^05:EmP1:
UNh0DYGf04ZEkP1BUOH0BYG^05:EmP1:UNh0DYGf04ZEkP1BUOH0BYG^00<0000000<0Y;kV0:C2kP2T
_^H00`0000002@2;h_H0U>Ko08_RmP2Di_l0Rn;f09CVo`2;h_H0U>Ko08_RmP020000000>0;C:kP2/
b^H0]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V0;C:kP00000?08>fmP0T0000
002L^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0U;K^09bj
kP2D]^h0W;[^09BfkP000000NjG^07^ZmP1kYNh0Nj[f07^UkP1kZ_H0NjG^07^ZmP1kYNh0Nj[f07^U
kP1kZ_H0NjG^07^ZmP1kYNh000000P3oool3000000P0oooo00<000000?ooo`3oool03P3oool00080
oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000030?ooo`@00000103oool00`000000
oooo0?ooo`0=0?ooo`0<0000002me^h0_M;^0;gFkP2md^h0_MK^0;gBkP2me^h0_M;^0;gFkP2md^h0
_MK^0`000000:02me^h0_M;^0;gFkP2md^h0_MK^0;gBkP2me^h0_M;^0;gFkP000000cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@000000]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0;C6iP2d
`^H0]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0;C6iP2d`^H2000000100;f^
e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^
c@2m[]D0]:k=0000002LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bI
e@2LULd0W9WE09bEc@2LVMD0W9G=0000001ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5
e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@000000BYG^05:ImP1:UNh0DYWf04ZEkP1BVOH0BYG^05:I
mP1:UNh0DYWf04ZEkP<0000000T0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h01000
00000`2Di_l0RnKo09CVo`020000000b0;C:kP2dc^h0]<[^0;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^
0;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^0;C>kP000000Rk[f08>fmP2;^_H0PkKf08^jmP23]_H0Rk[f
08>fmP2;^_H0PkKf08^jmP23]_H0Rk[f08>fmP2;^_H0000009bjkP2L]^h0W;[^09bfkP2L^^h0W;K^
09bjkP2L]^h0W;[^09bfkP2L^^h0W;K^09bjkP2L]^h0W;[^09bfkP2L^^h20000000>07^ZkP1kZ_H0
Nj[^07^ZmP1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^ZmP1kZ^h0Nj[f07^ZkP1kZ_H3000000`0oooo00<0
00000?ooo`3oool03@3oool000050?ooo`000000oooo0?ooo`000000203oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`0>0?ooo`80000000L0_M;V0;gFkP2md^H0_MK^0;gBiP2me^h0_M;V0080000002T0_MK^
0;gBiP2me^h0_M;V0;gFkP2md^H0_MK^0;gBiP2me^h0_M;V0;gFkP000000cLkN0<g:e@3=c]h0cL[E
0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0000002/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V
0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP02000000030:bbe@2d/]h00000
0180]:k=02T0000009bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=
09bEc@2LVLd0W9G=09bIc@000000LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE
07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE0000001:UNh0DYGf04ZEkP1BUOH0BYG^05:EmP040000000@
0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV
0:C2kP<0000001@0]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H0
]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H0]<[^000000l0PkKf02<0000009bjkP2D]^h0W;[^09BfkP2L
^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0000007^ZmP1k
YNh0Nj[f07^UkP1kZ_H0NjG^07^ZmP1kYNh0Nj[f07^UkP1kZ_H0NjG^07^ZmP1kYNh0Nj[f0000000=
0?ooo`030000003oool0oooo00d0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00<0oooo
00<000000?ooo`3oool04@3oool2000000030;gFkP2md^h000000080000000/0_M;^0;gFkP2md^h0
_MK^0;gBkP2me^h0_M;^0;gFkP2md^h0_MK^0;gBkP020000000<0<g>gP3=b]D0cLkN0<g:e@3=c]h0
cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0P0000003P2da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2d
a^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0P000000?@2d/]h0[;;E0;BbgP2//]D000000;B^
c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^
e@2d[/d0_JkE0000002LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bE
c@2LVMD0W9G=09bIe@2LULd0W9WE0000001cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5
gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0000004ZEkP1BVOH00`000000B`2/`^h0Y;k^0:c2
kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:Bn
kP2/`^h0Y;k^0000003=a]D000000;C:kP2dc^h0]<[^0;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^0;C>
kP2db^h0]<k^0;C:kP2dc^h0]<[^0;C>kP2db^h0]<k^0;C:kP2dc^h0000008^jmP23]_H0Rk[f08>f
mP2;^_H0PkKf08^jmP23]_H0Rk[f08>fmP2;^_H0PkKf08^jmP23]_H0Rk[f0000002L^^h0W;K^09bj
kP2L]^h0W;[^09bfkP2L^^h0W;K^09bjkP2L]^h0W;[^09bfkP2L^^h0W;K^09bjkP050000000=07^Z
mP1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^07^ZmP1kZ^h0Nj[f07^ZkP1kZ_H0Nj[^0000000=0?ooo`030000
003oool0oooo00d0oooo00001@3oool000000?ooo`3oool0000000P0oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo0080oooo0P00000C0?oo
o`<0000002X0_M;V0;gFkP2md^H0_MK^0;gBiP2me^h0_M;V0;gFkP2md^H0_MK^0;gBiP2me^h0_M;V
0000003=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN
0<g:e@3=c]h000000;C2iP2/`^H0]<;V0:c2iP2d`^H0[<;V0;C2iP2/`^H0]<;V0:c2iP2d`^H20000
00080:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@00000B0;B^c@0R0000002LVLd0W9G=09bIc@2L
ULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0000006Z5e@1c
QMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP<0
000001D0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:Bn
iP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h00P0000006@3=a]D0cL[E0<g6e@000000[<[V0;C:
kP2/b^H0]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H0]<[^0:c:
iP2db^h0[<[V0;C:kP0000003P23]_H02`000000U;K^09bjkP2D]^h0W;[^09BfkP2L^^h0U;K^09bj
kP2D]^h0W;[^00D0000000D0W9G=09bAa@000000NhWE08>=e@060000000807^UkP1kZ_H0NjG^07^Z
mP1kYNh0Nj[f07^UkP00000<0?ooo`030000003oool0oooo00d0oooo00001@3oool000000?ooo`3o
ool0000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`04
0000003oool0oooo000000<0oooo00<000000?ooo`3oool03`3oool3000000<0oooo02P000000;gB
kP2me^h0_M;^0;gFkP2md^h0_MK^0;gBkP2me^h0_M;^0;gFkP2md^h000000<g:e@3=c]h0cL[E0<g>
gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0000
002da^H0]<;V0;C6iP2d`^H0]<KV0;C2iP2da^H0]<;V0P000000>`2d/]h0[;;E0;BbgP2//]D0];;N
0:bbe@2d/]h0[;;E0;BbgP2//]D000000;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=
0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0000009bEc@2LVMD0W9G=09bIe@2LULd0W9WE
09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0000006Z5e@1cQMh0JXGE
07>5gP1ZQMD0LhGN06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN00<000000380Y:7E0:BMc@000000Y;k^
0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^
0:BnkP2/`^h0Y;k^0:c2kP000000cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D000000;C:kP2dc^h0]<[^
0;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^0;C>kP2db^h0]<k^0;C:kP2dc^h0]<[^0;C>kP2db^h0]<k^
0;C:kPT0000000/0PkKf08^jmP23]_H0Rk[f08>fmP2;^_H0PkKf0000002L]^h0W;[^09bfkP060000
000@09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0000007^=e@23SMh0NhgE08>=gP1kSMD0
PhgN07^=e@H0000000<0Nj[f0000003oool02`3oool00`000000oooo0?ooo`0=0?ooo`000P3oool2
000000X0oooo0P0000040?ooo`800000103oool2000000@0oooo00<000000?ooo`3oool0303oool3
000000L0oooo0P0000008`2me^h0_M;V0;gFkP2md^H0_MK^0;gBiP2me^h0_M;V0000003=b]D0cLkN
0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E
0<g>gP3=b]D0cLkN0000002d`^H0[<;V0;C2iP2/`^H0]<;V0080000000d0[;;E0;BbgP2//]D0];;N
0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h000000180]:k=01/0000009bEc@2LVLd0W9G=
09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@000000LhGN
06Z5e@1cQMh0JXGE07>5gP1ZQMD0LhGN06Z5e@1cQMh00`000000<`2TXMD0Y9g=0:BQe@2TWLd0Y:7E
0:BMc@000000Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV
0:C2kP2T_^H0Y<;^0:BniP2T`^h000000<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D00000
0:c:iP2db^h0[<[V0;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V
0;C:kP02000000090:BMc@000000RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG500/0000000h0W975
09bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@80000000`0
NhWE08>=e@1kRMD0PhgE07^9e@23SMD0NhWE08>=e@1kRMD0PhgE07^9e@23SMD3000000/0oooo00<0
00000?ooo`3oool03@3oool00200oooo00<000000?ooo`3oool02@3oool3000000`0oooo0P000000
102md^h0_MK^0;gBkP2me^h20000000I0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E
0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP000000]<KV
0;C2iP020000000i0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E
0;BbgP2//]D0];;N0000002m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE
0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0000002LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=
09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE0000001cQMh0JXGE07>5gP1ZQMD0LhGN
00<0000003<0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0000002T_^h0[<;^
0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/`^h00000
0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@000000]<[^0;C>kP2db^h0]<k^
0;C:kP2dc^h0]<[^0;C>kP2db^h0]<k^0;C:kP2dc^h00`0000009@2/XLd0[9g=0:bQc@2/WLd00000
08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd00000
09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=
09bAc@2LULd00P0000003`1kSMD0PhgN07^=e@23SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh0NhgE
08>=gP1kSMD0PhgN0000000:0?ooo`030000003oool0oooo00d0oooo000P0?ooo`030000003oool0
oooo00H0oooo0`00000A0?ooo`80000001/0_MK^0000003=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=
b]D0cLkN0<g:e@3=c]h00P0000004`2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d
/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP0000004@2d[/d050000000W9W=09bEc@2L
VLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=0000001Z
QMD0LhGN0`000000=02TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BM
c@2TXMD0Y9g=0000002T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0:C2
kP2T_^H0Y<;^0000003=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6
e@000000[<[V0;C:kP2/b^H0]<[^0:c:iP2db^h0[<[V0;C:kP2/b^H20000000X0:bQc@2TWLd0[:7=
0:BMc@2/XLd0Y9g=0:bQc@000000RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG5
08^9c@2;QLD0RhW=08^5a@2;RLd0000009bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=
09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@80000000l0PhgE07^9e@23SMD0NhWE08>=e@1kRMD0
PhgE07^9e@23SMD0NhWE08>=e@1kRMD0PhgE07^9e@0000002`3oool00`000000oooo0?ooo`0<0?oo
o`007`3oool2000000H0oooo0P00000F0?ooo`0I0000003=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=
b]D0cLkN0<g:e@020000000g0<g>gP000000];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d
/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D000000;B^c@2m[]D0]:k=0;f^e@2d
[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0000009bEc@2L
VMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@03
0000000N0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2T
WLd0Y:7E0:BMc@2TXMD0Y9g=0000002T_^h0[<;^0:BnkP2/`^h0Y;k^0:c2kP2T_^h0[<;^0:BnkP2/
`^h0Y;k^0P000000503=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6
e@3=b]D0cLKE0000002db^h0]<k^0;C:kP2dc^h0]<[^0`000000:`2/WLd0[:7=0:bMc@2/XLd0[9g=
0:bQc@2/WLd0[:7=0:bMc@000000RhG509B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=
08^5a@2DRLd0RhG509B9c@2;QLD0000009bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=
09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd00P0000003`23SMh0NhgE08>=gP1kSMD0PhgN
07^=e@23SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh0NhgE0000000:0?ooo`030000003oool0oooo
00`0oooo000O0?ooo`030000003oool0oooo0080oooo0`00000I0?ooo`80000002h0cL[E0<g>gP3=
b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=
c]h0cL[E0<g>gP3=b]D000000<G:gP35c]h0aL[N0<G>gP000000];;N0:bbe@2d/]h0[;;E0;BbgP2/
/]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D000004@2d[/d0
3@000000W9G=09bIc@2LULd0W9W=09bEc@2LVLd0W9G=09bIc@2LULd0W9W=09bEc@2LVLd00`000000
=@2TWLd0Y9W=0000002TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQ
e@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@000000Y<;^0:BniP2T`^h0Y;kV0:C2kP2T_^H0Y<;^0:Bn
iP2T`^h0Y;kV0000003=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:
e@3=a]D0cL[E0<g6e@3=b]D0cLKE0000002/b^H0]<[^008000000400Y9g=0:bQc@2TWLd0[:7=0:BM
c@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0000002;QLD0RhW=08^5a@2;RLd0RhG508^9
c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@000000W97509bEc@2LTLD0W9G=09bA
a@2LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=00000023SMD0000008>=
e@1kRMD0PhgE07^9e@23SMD0NhWE08>=e@1kRMD0PhgE07^9e@23SMD0NhWE08>=e@1kRMD000002@3o
ool00`000000oooo0?ooo`0<0?ooo`007`3oool00`000000oooo00000002000001h0oooo04T00000
0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E
0<g>gP3=b]D0cLkN0<g:e@000000aL[N0<g>gP35b]h0cLkN0<G:gP000000[;;E0;BbgP2//]D0];;N
0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h00000
0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE
0;B^c@2m[]D0000009bIe@2LULd0W9WE09bEc@2LVMD0W9G=09bIe@2LULd0W9WE00<0000003H0Y9g=
0:BIc@2TWLd0Y9W=0:BMc@000000Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=
0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0000002/`^h0Y;k^0:c2kP2T_^h0[<;^
0:BnkP2/`^h0Y;k^0000003=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE
0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D2000000120:bQc@2/WLd0[:7=0:bMc@2/XLd0
[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0000002DRLd0RhG509B9c@2;QLD0
U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@000000W9G=09bAc@2LULd0
W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@000000
PhgN00000023SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh0
00002@3oool00`000000oooo0?ooo`0<0?ooo`007`3oool200000280oooo0P0000003`3=c]h0cL[E
0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP020000000K
0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP000000[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E
0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h000000140]:k=00H00000
09bIc@2LULd0W9W=09bEc@2LVLd30000000h0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0
Y9g=0000002TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0
Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD000000:C2kP2T_^H0Y<;^0:BniP2T`^h0Y;kV0000003=a]D0
cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0
cLKE0<g:e@3=a]D2000000140:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/
XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0000008^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;
QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0000009bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@2L
ULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0000007^9e@000000NhWE08>=e@1k
RMD0PhgE07^9e@23SMD0NhWE08>=e@1kRMD0PhgE07^9e@23SMD0NhWE08>=e@0000080?ooo`030000
003oool0oooo00`0oooo000P0?ooo`030000003oool0oooo0280oooo040000000<g>gP3=b]D0cLkN
0<g:e@3=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP000000cLkN0<G:gP3=c]h0aL[N
0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0000002//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N
0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP000000_JkE0;B^c@2m[]D0]:k=
0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=0000002LULd0W9WE
0`0000009@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@000000
Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0
Y:7E0:BMc@2TXMD0Y9g=0:BQe@000000[<;^0:BnkP2/`^h00P000000FP3=a]D0cL[E0<g6e@3=b]D0
cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0
00000<g:c@3Eb]D000000:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0
[9g=0:bQc@2/WLd0[:7=0:bMc@000000RhG509B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0
U8W=08^5a@2DRLd0RhG509B9c@2;QLD0000009bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0
W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0000007^=e@000000NhgE08>=gP1kSMD0
PhgN07^=e@23SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh0NhgE000000P0oooo00<000000?ooo`3o
ool0303oool00240oooo0P00000S0?ooo`0;0000003=c]h0cL[E0<g>gP3=b]D0cLkN0<g:e@3=c]h0
cL[E0<g>gP3=b]D00P0000007`35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0
aLkN0<G:gP000000[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0
[;;E0;BbgP2//]D0];;N0:bbe@000000402d[/d3000000250:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=
0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@000000Y9g=0:BQe@2TWLd0Y:7E
0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=
0:BQe@000000Y<;^0000003=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E
0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0000003=a/d0eL[E0<g6c@000000Y9g=
0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=
0000002;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG5
08^9c@000000W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=
09bAa@2LULd0W97509bEc@000000PhgE07^9e@000000NhWE08>=e@1kRMD0PhgE07^9e@23SMD0NhWE
08>=e@1kRMD0PhgE07^9e@23SMD0NhWE000000070?ooo`030000003oool0oooo00`0oooo000S0?oo
o`030000003oool0oooo0240oooo0P000000=@3=b]D0cLkN0<g:e@3=c]h0cL[E0<g>gP3=b]D00000
0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N
0000002//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N
0:bbe@2d/]h0[;;E0000002d[/d0_JkE0;B^c@2m[]D0]:k=0;f^e@2d[/d0_JkE0;B^c@2m[]D0]:k=
00@0000008T0[:[N0000002TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=
0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0000002TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=
0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@000000cL[E
0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE
0<g:e@3=a]D0cL[E0<g6e@000000cL[=0=G:e@3=b/d0eL[E0<g:c@000000[9g=0:bQc@2/WLd0[:7=
0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0000002DRLd0RhG5
09B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@000000W9G=
09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=
09bEc@000000PhgN00000023SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh0NhgE08>=gP1kSMD0PhgN
07^=e@23SMh0NhgE000000070?ooo`030000003oool0oooo00/0oooo000T0?ooo`030000003oool0
oooo0280oooo02/000000<g:e@3=c]h0cL[E0<g>gP3=b]D000000<G:gP35c]h0aL[N0<G>gP35b]h0
aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP000000];;N0:bbe@2d/]h0
[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0
000000H0]:k=100000060:BUe@270000002TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BM
c@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0000002TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BM
c@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD000000<g2
e@000000cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6
e@3=b]D0cLKE0<g:e@3=a]D000000<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0000002/XLd0Y9g=0:bQ
c@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0000008^5
a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG50000
002LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=09bA
a@2LULd0W9750000001kRMD0PhgE00000023SMD0NhWE08>=e@1kRMD0PhgE07^9e@23SMD0NhWE08>=
e@1kRMD0PhgE07^9e@23SMD0000000L0oooo00<000000?ooo`3oool02`3oool002D0oooo00<00000
0?ooo`3oool08P3oool20000000Z0<g>gP00000000000<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN0<G:
gP3=c]h0aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN0000002d/]h0[;;E0;Bb
gP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0000
002m[]D0]:k=10000000TP2/Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD00000
0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=
0:BMc@2TVLd000000:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E
0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0000003=`]D0cL;=0<g2e@000000cL[E0<g6e@3=b]D0cLKE
0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0000003=b/d0eL[E
0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0000002/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=
0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@000000U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=
08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd0000009bEc@2LTLd0W9G=09bAc@2LULd0W97=
09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=0000001kSMD0PhgN
00000023SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh0NhgE08>=gP1kSMD0PhgN07^=e@23SMh00000
1P3oool00`000000oooo0?ooo`0;0?ooo`009P3oool2000002@0oooo02L000000<G>gP35b]h0aLkN
0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N
0<G>gP35b]h0aLkN0000002d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N
0:bbe@2d/]h0[;;E0;BbgP2//]D00`00000?0:BUe@0R0000002TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BM
c@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0000002TWLd0Y:7E0:BM
c@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@80000006<0cL;E
0<G2c@3=`]D0aL;=0<g2e@000000cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE
0<g:e@3=a]D0cL[E0<g6e@000000cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@3=a/d00000
0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=
0:bQc@000000RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=
08^5a@2;RLd0000009bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W975
09bEc@2LTLD0W9G=09bAa@2LULd0000008>=e@1kRMD0PhgE00000023SMD0NhWE08>=e@1kRMD0PhgE
07^9e@23SMD0NhWE08>=e@1kRMD0PhgE07^9e@23SMD00P0000040?ooo`030000003oool0oooo00/0
oooo000X0?ooo`030000003oool0oooo0280oooo02<000000<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N
0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN
0000002d/]h0[;;E0;BbgP2//]D0];;N0:bbe@2d/]h0[;;E0;BbgP2//]D0];;N0:bbe@030000002K
0:B^gP2//]h000000:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE
0:bZgP2TYMD0[:[N0000002TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=
0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0000002TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=
0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0000003=`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E
0000003=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@000000cL[=
0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@000000[9g=0:bQc@2/WLd0[:7=
0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@2/WLd0000008^5a@2DRLd0RhG5
09B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd0RhG50000002LTLd0W9G=
09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=
09bEc@000000PhgN07^=e@000000NhgE08>=gP1kSMD0PhgN07^=e@23SMh0NhgE08>=gP1kSMD0PhgN
07^=e@23SMh0NhgE0000001B00000`0000000`3oool000000?ooo`0<0?ooo`00:@3oool00`000000
oooo0?ooo`0R0?ooo`80000001h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN
0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP000000];;N0:bbe@2d/]h0[;;E
0;BbgP2//]D0];;N0:bbe@2d/]h2000000D0Y:kN00<000000:BUe@2TYMD03P2TYMD0P`000000Y9g=
0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=
0:BMc@000000Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd00000
0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@000000cL[E0<g6e@3=b]D0cLKE
0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0000003=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E
0<g6c@3Eb]D0cLK=0=G:e@000000[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=
0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0000002;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=
08^5a@2;RLd0RhG508^9c@2;QLD0RhW=0000002LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W975
09bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@000000NhWE08>=e@1kRMD00000
07^9e@23SMD0NhWE08>=e@1kRMD0PhgE07^9e@23SMD0NhWE00X000003@3oool002X0oooo00<00000
0?ooo`3oool08`3oool06`000000aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN
0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0000002//]D0];;N0:bbe@2d/]h0[;;E
0;BbgP030000002B0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h000000:BUe@2/Z]h0Y:GE
0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0000002TVLd0Y9g=
0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=
0000002TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@2TXMD0Y9g=0000003=`/d0cL;E
0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0000003=a]D0cL[E0<g6e@3=b]D0cLKE
0<g:e@3=a]D0cL[E0<g6e@000000cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E
0<g:c@3Eb]D0cL[=0000002/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=
0:bQc@2/WLd0[:7=0:bMc@2/XLd0000009B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=
08^5a@2DRLd0RhG509B9c@2;QLD0U8W=0000002LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=
09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@000000NhgE08>=gP1kSMD90000
00T0oooo00<000000?ooo`3oool02`3oool002/0oooo00<000000?ooo`3oool08`3oool05`000000
aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0
aLkN0<G:gP35c]h0aL[N0000002//]D0];;N00<00000302T[]h00`000000Y:GE0:BUe@0=0:BUe@1d
0000002TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=
0:BIc@2TWLd0Y9W=0000002TWLd0Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@000000aL;=
0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0000003=a]D0cL[E
0<g6e@3=b]D0cLKE0<g:e@3=a]D000000<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=
0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0000002TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=
0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@000000RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG5
08^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0000009bAa@2LULd0W97509bEc@2LTLD0W9G=
09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD700000140oooo00<00000
0?ooo`3oool02P3oool002`0oooo0P00000T0?ooo`8000000140cLkN0<G:gP3=c]h0aL[N0<g>gP35
b]h0cLkN0<G:gP3=c]h0aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N0<g>gP030000002A0:bbgP2T
[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N0000002/
Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP00
0000Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2T
WLd0Y9W=0:BMc@000000Y:7E0:BMc@2TXMD0Y9g=0:BQe@2TWLd0Y:7E0:BMc@000000cL;=0<g2e@3=
`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d000000<g6e@3=
b]D0cLKE0<g:e@3=a]D000000<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=
b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@000000[:7=0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/
XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0000002;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;
QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@000000W97=09bEc@2LTLd0W9G=09bAc@2L
ULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd0W97=09bEc@02000000H0oooo00<000000?ooo`3oool0
3`3oool00`000000oooo0?ooo`0:0?ooo`00;P3oool00`000000oooo0?ooo`0S0?ooo`0=00000035
c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP04000000030:C:mP2T
a^h000000100Y:kN00<000000:BUe@2TYMD03@2TYMD0K`000000Y9g=0:BIc@2TWLd0Y9W=0:BMc@2T
VLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@000000Y:7E0:BMc@2T
XMD0Y9g=0:BQe@2TWLd000000<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35
`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d000000<g6e@3=b]D0cLKE0000003=a/d0eL[E0<g6c@3E
b]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@00
0000[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BMc@2/
XLd0000008^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@2;
QLD0RhW=0000002LULd0W97509bEc@2LTLD0W9G=09bAa@2LULd0W97509bEc@2LTLD0W9G=09bAa@02
000000T0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0:0?ooo`00;`3oool00`00
0000oooo0?ooo`0S0?ooo`090000003=c]h0aL[N0<g>gP35b]h0cLkN0<G:gP3=c]h0aL[N00<00000
03h0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0Y<K^0000002T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T
[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N0:B^gP2//]h000000:bZgP2TYMD0[:[N0:BUe@2/
Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD000000:BIc@2TWLd0Y9W=0:BMc@2T
VLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd000000:BMc@2T
XMD0Y9g=0:BQe@80000005@0cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E0<g2
c@3=`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0000003=a]D000000<g:c@3Eb]D0cL[=0=G:
e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:
c@000000[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQ
c@2/WLd0000008^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5
a@2DRLd0RhG50000002LTLd0W9G=09bAc@2LULd0W97=09bEc@2LTLd0W9G=09bAc@2LULd2000000`0
oooo00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0:0?ooo`00<03oool00`000000oooo
0?ooo`0S0?ooo`80000000<0aL[N0<G>gP0000000`0000002`2Ta^h0Y<[f0:C6kP2Tb_H0Y<K^0:C:
mP2Ta^h0Y<[f0:C6kP2Tb_H000000100Y:kN00<000000:BUe@2TYMD03@2TYMD0J@000000Y9W=0:BM
c@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BI
c@000000Y9g=0:BQe@000000cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2
e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0000003=a/d0eL[E0<g6
c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:
e@3=a/d0eL[E0<g6c@000000Y9g=0:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0:BM
c@2/XLd0Y9g=0:bQc@2TWLd0000008^5a@2;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9
c@2;QLD0RhW=08^5a@2;RLd0RhG50000002LTLD0W9G=09bAa@2LULd0W97509bEc@03000000h0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`0:0?ooo`00<@3oool2000002D0oooo0P00
0000UP2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0Y<K^
0:c:mP000000[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N
0:B^gP2//]h0Y:kN0000002TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE
0:bZgP2TYMD0[:[N0000002TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=
0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0000002TXMD000000<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=
0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E
0<g2c@000000g_KN0000003=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=
0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0000002/XLd0[9g=0:bQc@2/WLd0[:7=
0:bMc@2/XLd0[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@000000U8W=08^5a@2DRLd0RhG5
09B9c@2;QLD0U8W=08^5a@2DRLd0RhG509B9c@2;QLD0U8W=08^5a@2DRLd0000009bEc@2LTLd0W9G=
09bAc@8000004P3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo00X0oooo000c0?oo
o`030000003oool0oooo02<0oooo018000000:C6kP2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP2Tb_H0
Y<K^0:C:mP2Ta^h0Y<[f0:C6kP2Tb_H0Y<K^0:C:mP00000@0:B^gP030000002TYMD0Y:GE00`0Y:GE
014000000:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2T
VLd0Y9g=0:BIc@020000001A0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=
`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@000000eO;N0=kfgP3E
l]h000000<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3E
b]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D000000:bQc@2TWLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2T
WLd0[:7=0:BMc@2/XLd0Y9g=0:bQc@2TWLd0[:7=0000002;RLd0RhG508^9c@2;QLD0RhW=08^5a@2;
RLd0RhG508^9c@2;QLD0RhW=08^5a@2;RLd0RhG508^9c@000000W9G=008000005@3oool00`000000
oooo0?ooo`0;0?ooo`030000003oool0oooo00X0oooo000d0?ooo`030000003oool0oooo02<0oooo
0P000000?@2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0
Y<K^0:c:mP000000[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0
[;;N0:B^gP2//]h0Y:kN0000002TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0
Y:GE0:bZgP2TYMD000000:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@2TVLd0
Y9g=0:BIc@2TWLd00P000000602/_^H0];kV0000003=`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0
cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=000000D0g_KN
024000000<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@3E
b]D0cL[=0=G:e@3=b/d0eL[E0<g:c@000000[9g=0:bQc@2/WLd0[:7=0:bMc@2/XLd0[9g=0:bQc@2/
WLd0[:7=0:bMc@2/XLd05@00000H0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool0
2@3oool003D0oooo00<000000?ooo`3oool0903oool040000000Y<[f0:C6kP2Tb_H0Y<K^0:C:mP2T
a^h0Y<[f0:C6kP2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP00000@0:B^gP030000002TYMD0Y:GE00`0
Y:GE00/000000:BIc@2TWLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0:BMc@030000000h0;Bn
iP2/_]h0];kV0:bngP2d_^H000000<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2
c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0000003El]h0g_KN0=GbgP3Nm]h0eO;N0=kf
gP3El]h000000<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6
c@3Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@000000Y9g=0:bQc@2TWLd8000002d0oooo00<000000?oo
o`3oool02`3oool00`000000oooo0?ooo`090?ooo`00=P3oool2000002D0oooo03L000000:c:mP2T
a^h0[<[f0:C6kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0Y<K^0:c:mP2Ta^h000000:B^gP2/
/]h0Y:kN0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP00
0000[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N0000002T
WLd0Y9W=0:BMc@2TVLd0Y9g=0:BIc@2TWLd0Y9W=0080000001/0];kV0:bniP2d_^H0[;kV0;BniP2/
_^H0];kV0:bniP000000cL;=0<g2e@3=`/d0cL;E0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@3=
`]D0cL;=0<g2e@3=`/d0cL;E0<g2c@000000203Nm]h04`000000eL[E0<g:c@3Eb]D0cL[=0=G:e@3=
b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d0eL[E0<g:c@3Eb]D0cL[=0=G:e@3=b/d01@00000f0?oo
o`030000003oool0oooo00X0oooo00<000000?ooo`3oool02@3oool003P0oooo00<000000?ooo`3o
ool08`3oool20000000>0:C6kP2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP2Tb_H0Y<K^0:C:mP2Ta^h0
Y<[f0:C6kP00000@0:B^gP030000002TYMD0Y:GE00/0Y:GE00D000000:BMc@2TVLd0Y9g=0:BIc@03
0000000e0:bngP2d_^H0[;kN0;BniP2/_]h0];kV0:bngP2d_^H0[;kN0;BniP2/_]h000000<G2c@3=
`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@35`/d0cL;E0<G2c@3=`]D0aL;=0000003E
l]h0g_KN0=GbgP3Nm]h0eO;N0=kfgP3El]h0g_KN0=GbgP3Nm]h000000=G:e@3=a/d0eL[E0<g6c@3E
b]D0cLK=0=G:e@3=a/d0eL[E0<g6c@3Eb]D0cLK=0=G:e@3=a/d00`000000102/`^H0[<;^0:c2iP00
000h0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool02@3oool003T0oooo00<00000
0?ooo`3oool0903oool0;`000000Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f
0:C6kP2/b_H0Y<K^0000002T[]h0[;;N0:B^gP2//]h0Y:kN0:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN
0:bbgP2T[]h0[;;N0:B^gP000000Y:GE0:bZgP2TYMD0[:[N0:BUe@2/Z]h0Y:GE0:bZgP2TYMD0[:[N
0:BUe@2/Z]h0Y:GE0000002TVLd0Y9g=0080000001`0[;kV0;BniP2/_^H0];kV0:bniP2d_^H0[;kV
0;BniP2/_^H0];kV0:bniP2d_^H0[;kV0;BniP2/_^H000000<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E
0<g2c@3=`]D0cL;=0<g2e@3=`/d0cL;E0P00000<0=kfgP0;0000003Eb]D0cL[=0=G:e@3=b/d0eL[E
0<g:c@3Eb]D0cL[=0=G:e@3=b/d00`0000001`2/`^H0[<;^0:c2iP2/`^h0[<;V0:c2kP000000>03o
ool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo000j0?ooo`030000003oool0
oooo02@0oooo00l000000:C6kP2Tb_H0Y<K^0:C:mP2Ta^h0Y<[f0:C6kP2Tb_H0Y<K^0:C:mP2Ta^h0
Y<[f0:C6kP0000003@2T[]hB0000000d0;BniP2/_]h0];kV0:bngP2d_^H0[;kN0;BniP2/_]h0];kV
0:bngP2d_^H0[;kN0;BniP2/_]h0];kV0:bngP2d_^H0[;kN00000035`/d0cL;E0<G2c@3=`]D0aL;=
0<g2e@35`/d0cL;E0<G2c@3=`]D000000=kfgP3El]h0g_KN0=GbgP3Nm]h0eO;N0=kfgP3El]h0g_KN
0=GbgP3Nm]h0eO;N0=kfgP3El]h0g_KN0000003Eb]D0cLK=0=G:e@3=a/d0eL[E0<g6c@<0000000T0
[<;V0:c2kP2/`^H0[<;^0:c2iP2/`^h0[<;V0:c2kP000000>P3oool00`000000oooo0?ooo`080?oo
o`030000003oool0oooo00T0oooo000k0?ooo`8000009@3oool20000000E0:c:mP2Ta^h0[<[f0:C6
kP2/b_H0Y<K^0:c:mP2Ta^h0[<[f0:C6kP2/b_H000000:bbgP2T[]h0[;;N0:B^gP2//]h0Y:kN0:bb
gP2T[]h0[;;N00@0000002l0U;[^09bnmP2D^^h0000007^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^b
mP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP000000[;kV0;BniP2/_^H0];kV0:bniP2d_^H0[;kV0;Bn
iP2/_^H0];kV0:bniP2d_^H0[;kV0;BniP2/_^H0];kV0:bniP2d_^H0[;kV0000003=`/d0cL;E0<g2
c@3=`]D0cL;=0<g2e@3=`/d0cL;E0000000A0=kfgP030000003Eb]D0cL[=00<0000000/0[<;V0:c2
kP2/`^H0[<;^0:c2iP2/`^h0[<;V0:c2kP2/`^H0[<;^0000000l0?ooo`030000003oool0oooo00L0
oooo00<000000?ooo`3oool02@3oool003d0oooo00<000000?ooo`3oool0903oool7000000060:C:
mP2Ta^h0Y<[f0:C6kP2Tb_H000000`2T[]h50000001409BnmP2D^^h0U;kf09BjkP2D__H0U;[^09Bn
mP2D^^h0000007^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^b
mP000000[;kN0;BniP2/_]h0];kV0:bngP2d_^H0[;kN0;BniP2/_]h0];kV0:bngP2d_^H0[;kN0;Bn
iP2/_]h0];kV0:bngP2d_^H0[;kN00000035`/d0cL;E0<G2c@3=`]D0aL;=0<g2e@000000g_KN0=Gb
gP3Nm]h0eO;N0=kfgP3El]h0g_KN0=GbgP3Nm]h0eO;N0=kfgP3El]h0g_KN0=GbgP3Nm]h0eO;N0=kf
gP3El]h30000000>0:c2iP2/`^h0[<;V0:c2kP2/`^H0[<;^0:c2iP2/`^h0[<;V0:c2kP2/`^H0[<;^
0:c2iP00000m0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool0203oool003h0oooo
00<000000?ooo`3oool0903oool20000000405Z^mP1J[_l0FZkf05Z^o`T0000003H0W;kf09BjkP2L
__H0U;[^09bnmP2D^^h0W;kf09BjkP2L__H0U;[^09bnmP2D^^h0W;kf09BjkP000000Nk;f08>fmP1k
/_H0PkKf07^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP23]_H0Nk;f0000002/_^H0];kV0:bniP2d
_^H0[;kV0;BniP2/_^H0];kV0:bniP2d_^H0[;kV0;BniP2/_^H0];kV0:bniP2d_^H0[;kV0;BniP00
0000cL;E0<g2c@3=`]D0cL;=0<g2e@00000@0=kfgP<000000100Y8Vm0:B=_@000000[<;^0:c2iP2/
`^h0[<;V0:c2kP2/`^H0[<;^0:c2iP2/`^h0[<;V0:c2kP2/`^H00000?P3oool00`000000oooo0?oo
o`070?ooo`030000003oool0oooo00P0oooo000o0?ooo`030000003oool0oooo02D0oooo05400000
05:ZmP1J[_l0DZ[f05Z^o`1BZ_H0FZko05:ZmP1J[_l0DZ[f05Z^o`000000U;kf09BjkP2D__H0U;[^
09BnmP2D^^h0U;kf09BjkP2D__H0U;[^09BnmP2D^^h0U;kf09BjkP2D__H0U;[^0000001k/_H0PkKf
07^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP23]_H0Nk;f08>fmP1k/_H000000:bngP2d_^H0[;kN
0;BniP2/_]h0];kV0:bngP2d_^H0[;kN0;BniP2/_]h0];kV0:bngP2d_^H0[;kN0;BniP2/_]h0];kV
0000003=`]D0aL;=0<g2e@000000g_KN0=GbgP3Nm]h0eO;N0=kfgP3El]h0g_KN0=GbgP3Nm]h0eO;N
0=kfgP3El]h0g_KN0=GbgP3Nm]h00P0000004P2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@000000[<;^
0:c2iP2/`^h0[<;V0:c2kP2/`^H0[<;^0:c2iP2/`^h0[<;V00000400oooo00<000000?ooo`3oool0
1P3oool00`000000oooo0?ooo`080?ooo`00@03oool2000002H0oooo00/0000005Z^mP1J[_l0FZkf
05Z^o`1J[_H0FZko05Z^mP1J[_l0FZkf05Z^o`020000000?09BjkP2L__H0U;[^09bnmP2D^^h0W;kf
09BjkP2L__H0U;[^09bnmP2D^^h0W;kf09BjkP2L__H0U;[^008000000280PkKf07^bmP23]_H0Nk;f
08>fmP1k/_H0PkKf07^bmP23]_H0Nk;f08>fmP1k/_H000000:bniP2d_^H0[;kV0;BniP2/_^H0];kV
0:bniP2d_^H0[;kV0;BniP2/_^H0];kV0:bniP2d_^H0[;kV0;BniP2/_^H0];kV0000003=`]D00000
3@3Nm]h30000000D0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0000002/`^h0[<;V
0:c2kP2/`^H0[<;^0:c2iP2/`^h0[<;V0:c2kP0000110?ooo`030000003oool0oooo00D0oooo00<0
00000?ooo`3oool0203oool00480oooo00<000000?ooo`3oool0903oool20000000j05Z^o`1BZ_H0
FZko05:ZmP1J[_l0DZ[f05Z^o`1BZ_H0FZko05:ZmP000000U;[^09BnmP2D^^h0U;kf09BjkP2D__H0
U;[^09BnmP2D^^h0U;kf09BjkP2D__H0U;[^09BnmP2D^^h0U;kf00000023]_H0Nk;f08>fmP1k/_H0
PkKf07^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP000000[;kN0;BniP2/_]h0];kV0:bngP2d_^H0
[;kN0;BniP2/_]h0];kV0:bngP2d_^H0[;kN0;BniP2/_]h0];kV0:bngPH0000000L0g_KN0=GbgP3N
m]h0eO;N0=kfgP3El]h0g_KN00<0000001H0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2T
SKd0W8Vm0:B=_@2LRKd0Y8fm0000002/`^h0[<;V0:c2kP2/`^H0[<;^0:c2iP2/`^h00000@`3oool0
0`000000oooo0?ooo`040?ooo`030000003oool0oooo00P0oooo00130?ooo`030000003oool0oooo
02D0oooo03L0000005Z^o`1J[_H0FZko05Z^mP1J[_l0FZkf05Z^o`1J[_H0FZko05Z^mP000000U;[^
09bnmP2D^^h0W;kf09BjkP2L__H0U;[^09bnmP2D^^h0W;kf09BjkP2L__H0U;[^09bnmP2D^^h0W;kf
00000023]_H0Nk;f08>fmP1k/_H0PkKf07^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP000000[;kV
0;BniP2/_^H0];kV0:bniP2d_^H0[;kV0;BniP2/_^H0];kV0:bniP2d_^H0[;kV00<000001@1:a[@8
0000000I0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2T
RKd0Y8fm0:B9_@2TSKd000000:c2kP2/`^H0[<;^0:c2iP2/`^h0[<;V000000130?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool0203oool004@0oooo00<000000?ooo`3oool09@3oool2
0000000b05:ZmP1J[_l0DZ[f05Z^o`1BZ_H0FZko05:ZmP1J[_l0DZ[f0000002D^^h0U;kf09BjkP2D
__H0U;[^09BnmP2D^^h0U;kf09BjkP2D__H0U;[^09BnmP2D^^h0U;kf09BjkP2D__H0000008>fmP1k
/_H0PkKf07^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP23]_H0Nk;f0000002/_]h0];kV0:bngP2d
_^H0[;kN0;BniP2/_]h0];kV0:bngP2d_^H:0000000P09Ai[02LOK@0U7V/09am]02DNJ`0W7fd0000
002TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=
_@2LRKd0Y8fm09b9_@2TSKd000000:c2kP2/`^H0[<;^0:c2iP0000150?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool0203oool004D0oooo00<000000?ooo`3oool09P3oool30000000/
05Z^mP1J[_l0FZkf05Z^o`1J[_H0FZko05Z^mP000000U;[^09bnmP2D^^h0W;kf09BjkP2L__H0U;[^
09bnmP2D^^h0W;kf09BjkP2L__H0U;[^09bnmP2D^^h0W;kf00000023]_H0Nk;f08>fmP1k/_H0PkKf
07^bmP23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP000000[;kV0;BniP2/_^H0];kV0:bniP2d_^H40000
000X09Ai]0000000U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0
U7Vd09am]0000000Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0
Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0000002/`^h0[<;V000004L0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`070?ooo`00AP3oool2000002L0oooo1P000000101J[_l0DZ[f
05Z^o`1BZ_H20000000<09BnmP2D^^h0U;kf09BjkP2D__H0U;[^09BnmP2D^^h0U;kf09BjkP2D__H0
U;[^2P0000002@23]_H0Nk;f08>fmP1k/_H0PkKf07^bmP000000[;kN0;BniP04000000@0U7Vd02L0
000009am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0
000009b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0
W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd000000:c2kP000000A`3oool00`000000oooo0?ooo`030?oo
o`030000003oool0oooo00L0oooo00180?ooo`030000003oool0oooo02H0oooo0P0000000`0HTNH0
69G^000000040000000605Z^o`000000W;kf09BjkP2L__H0U;[^1`0000001@1Rd_H0J]Kf06;BmP1Z
e_H0H];f00<0000000<0Y8fm0:bA_@00000020000000;P2DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02D
OK@0U7Vd09Am]0000000W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02D
NK@0W7fd0000002TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2T
SKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@0000190?ooo`030000003oool0oooo0080
oooo00<000000?ooo`3oool01`3oool004T0oooo00<000000?ooo`3oool09`3oool20000000501RA
kP0HTNH0697^01RAiP0HTNh01P0000002P1Rd^h0H];f06;BkP1Rd_H0H];^06;BmP1Rd^h0H];f06;B
kP1Rd_H20000000:0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd000000d0U7Vd02@0
000009Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0
00000:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0
Y8fm09b9_@2TSKd0W8Vm0:B=_@00001;0?ooo`050000003oool0oooo0?ooo`0000002@3oool004X0
oooo00<000000?ooo`3oool0:03oool20000000801RAiP0HUNh0697V01REkP0HTNH069G^01RAiP0H
UNh50000000406[FmP1Rd_H0J]Kf06;BmP80000003d0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m
0:B=_@2/TKd0Y8fm0:bA_@000000U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd
09Am]02DNK@0U7fd0000002LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd
09Ai]02LOK@0U7Vd0000002TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm
0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0000004`0oooo00@000000?ooo`3oool00000
2@3oool004/0oooo0P00000Z0?ooo`80000000L0697^01RAiP0HTNh0697V01RAkP0HTNH0697^00P0
000000h0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0000
00d0U7Vd02<0000009Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd
09Ai[02LOK@000000:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm
0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0000001=0?ooo`040000003oool0oooo000000T0oooo001=
0?ooo`030000003oool0oooo02T0oooo1`000000AP2LTLd0U97=09bAc@2DTLd0W97=09BAc@000000
Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0
000009Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]0000000
U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]0000000
Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0
Y8Vm0:B=_@00001>0?ooo`030000003oool0000000T0oooo001>0?ooo`030000003oool0oooo02P0
oooo01h0000009B=c@2LTLd0U8g=09bAc@2DSLd0W97=09B=c@2LTLd0U8g=09bAc@2DSLd0W97=09B=
c@000000Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=
_@00000=09Ai]00Q0000002LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd
09Ai[02LOK@0U7V/0000002LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm
0:B=_@2LRKd0Y8fm09b9_@2TSKd000000500oooo00<000000?ooo`000000203oool004l0oooo00<0
00000?ooo`3oool0:03oool0BP000000U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0
W97=09BAc@2LTLd000000:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0
Y8fm0:bA_@2TSKd0000009Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0
U7fd09Ai]0000000U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0
U7Vd09am]0000000Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0
Y8fm0:B9_@2TSKd00000DP3oool2000000P0oooo001@0?ooo`800000:03oool07@000000W97=09B=
c@2LTLd0U8g=09bAc@2DSLd0W97=09B=c@2LTLd0U8g=09bAc@2DSLd000000:B=]02TSKd0Y8fd0:B=
_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0000000d0U7Vd0200000009am
]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd0000
002TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0000
0580oooo0P0000080?ooo`00DP3oool00`000000oooo0?ooo`0V0?ooo`180000002LTLd0U97=09bA
c@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@000000Y8fm0:bA_@2TSKd0[96m0:B=
_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@000000U7fd09Ai]02DOK@0U7Vd09Am
]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd0000002LOK@0U7Vd09am]02DNK@0W7fd09Ai
]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd0000002TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=
_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd00000D`3oool2000000P0oooo001C0?ooo`030000
003oool0oooo02D0oooo01d0000009B=c@2LTLd0U8g=09bAc@2DSLd0W97=09B=c@2LTLd0U8g=09bA
c@2DSLd0W97=0000002TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=
]02TSKd0Y8fd0000000=09Ai]00N0000002DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai
[02LOK@0U7V/09am]02DNJ`0W7fd0000002TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=
_@2LRKd0Y8fm09b9_@2TSKd00000C`3oool4000000X0oooo001D0?ooo`030000003oool0oooo02D0
oooo04H0000009BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=0000
002/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0000009Am
]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]0000000W7fd09Ai
]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@000000:B=
_@2TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@2TSKd00000C03oool4000000h0oooo
001E0?ooo`8000009@3oool070000000W97=09B=c@2LTLd0U8g=09bAc@2DSLd0W97=09B=c@2LTLd0
U8g=09bAc@2DSLd000000:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0
Y8fm0:B=]000000=09Ai]00L0000002DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02L
OK@0U7V/09am]02DNJ`0W7fd09Ai[0000000W8Vm0:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm0:B=_@2L
RKd0Y8fm000004T0oooo1000000B0?ooo`00E`3oool00`000000oooo0?ooo`0S0?ooo`140000002L
TLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=0000002/TKd0Y8fm0:bA_@2T
SKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0000002DNK@0U7fd09Ai]02D
OK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0000009Ai]02LOK@0U7Vd09am]02D
NK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@000000:B=_@2TRKd0Y8fm0:B9_@2T
SKd0Y8Vm0:B=_@2TRKd0Y8fm0:B9_@0000150?ooo`@000005P3oool005P0oooo00<000000?ooo`3o
ool08P3oool070000000U8g=09bAc@2DSLd0W97=09B=c@2LTLd0U8g=09bAc@2DSLd0W97=09B=c@2L
TLd000000:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@00
000=09Ai]00J0000002LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai
[02LOK@0U7V/09am]0000000Y8fm09b9_@2TSKd0W8Vm0:B=_@2LRKd0Y8fm09b9_@0000120?ooo`@0
00006P3oool005T0oooo00<000000?ooo`3oool08P3oool0@@000000U97=09bAc@2DTLd0W97=09BA
c@2LTLd0U97=09bAc@2DTLd0W97=09BAc@000000Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA
_@2TSKd0[96m0:B=_@2/TKd0Y8fm0000002DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai
]02DOK@0U7Vd09Am]02DNK@0000009Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am
]02DNK@0W7fd09Ai]02LOK@0U7Vd0000002TRKd0Y8fm0:B9_@2TSKd0Y8Vm0:B=_@2TRKd000000400
oooo0`00000N0?ooo`00FP3oool200000280oooo01/0000009bAc@2DSLd0W97=09B=c@2LTLd0U8g=
09bAc@2DSLd0W97=09B=c@2LTLd000000:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm
0:B=]02TSKd0Y8fd0:B=_@0000003@2DNK@06@000000W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd
09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@000000:B=_@2LRKd0Y8fm09b9_@2TSKd0W8Vm
0:B=_@000000?03oool400000240oooo001L0?ooo`030000003oool0oooo0200oooo03l0000009bA
c@2DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd000000:bA_@2TSKd0[96m0:B=
_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@000000U7fd09Ai]02DOK@0U7Vd09Am
]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd0000002LOK@0U7Vd09am]02DNK@0W7fd09Ai
]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]0000000Y8fm0:B9_@2TSKd0Y8Vm0:B=
_@000000>@3oool4000002D0oooo001M0?ooo`030000003oool0oooo01l0oooo01/0000009B=c@2L
TLd0U8g=09bAc@2DSLd0W97=09B=c@2LTLd0U8g=09bAc@2DSLd000000:B=]02TSKd0Y8fd0:B=_@2T
SK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]00000003@2DNK@05`000000U7V/09am]02D
NJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0000009b9_@2T
SKd0W8Vm0:B=_@2LRKd0000003D0oooo1000000Y0?ooo`00GP3oool00`000000oooo0?ooo`0O0?oo
o`0m0000002DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=0000002TSKd0
[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@000000U7fd09Ai]02DOK@0
U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd0000002LOK@0U7Vd09am]02DNK@0
W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]0000000Y8fm0:B9_@2TSKd0
Y8Vm0000000b0?ooo`@00000;@3oool005l0oooo00<000000?ooo`3oool07P3oool06P000000W97=
09B=c@2LTLd0U8g=09bAc@2DSLd0W97=09B=c@2LTLd0U8g=09bAc@000000Y8fm0:B=]02TSKd0Y8fd
0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@000003@2DNK@05@000000U7V/09am]02DNJ`0
W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd0000002TSKd0
W8Vm0000000`0?ooo`<00000<@3oool00600oooo0P00000O0?ooo`0k0000002LTLd0U97=09bAc@2D
TLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd000000:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/
TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@000000U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02D
NK@0U7fd09Ai]02DOK@0U7Vd0000002DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02L
OK@0U7Vd09am]02DNK@0W7fd09Ai]0000000Y8Vm0:B=_@000000;03oool4000003@0oooo001R0?oo
o`030000003oool0oooo01`0oooo01X0000009B=c@2LTLd0U8g=09bAc@2DSLd0W97=09B=c@2LTLd0
U8g=09bAc@2DSLd000000:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0
Y8fm000000d0U7Vd01<0000009am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02L
OK@0U7V/09am]02DNJ`0W7fd0000002TSKd0000002T0oooo1000000h0?ooo`00H`3oool00`000000
oooo0?ooo`0L0?ooo`0g0000002DTLd0W97=09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0
00000:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0000002DNK@0
U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0000009Ai]02LOK@0
U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]0020000
02D0oooo1000000l0?ooo`00I03oool00`000000oooo0?ooo`0K0?ooo`0I0000002LTLd0U8g=09bA
c@2DSLd0W97=09B=c@2LTLd0U8g=09bAc@2DSLd000000:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=
]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0000000=09Ai]00B0000002LOK@0U7V/09am]02DNJ`0W7fd09Ai
[02LOK@0U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`000008P3oool400000400oooo
001U0?ooo`800000703oool20000000d09BAc@2LTLd0U97=09bAc@2DTLd0W97=09BAc@2LTLd0U97=
0000002TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@000000U7fd
09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd0000002LOK@0U7Vd
09am]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]02DNK@0W7fd09Ai]02LOK@0U7Vd000001l0oooo1000
00140?ooo`00I`3oool00`000000oooo0?ooo`0I0?ooo`0I0000003oool000000000002LTLd0U8g=
09bAc@2DSLd0W97=09B=c@2LTLd000000:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm
0:B=]02TSKd0Y8fd0000000=09Ai]00=0000002DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd
09Ai[02LOK@0U7V/09am]002000001d0oooo0`0000180?ooo`00J03oool00`000000oooo0?ooo`0H
0?ooo`040000003oool0oooo0?ooo`80000002d0U97=09bAc@2DTLd0W97=09BAc@2LTLd000000:bA
_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0000009Am]02DNK@0U7fd09Ai
]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]0000000W7fd09Ai]02LOK@0U7Vd09am
]02DNK@0W7fd09Ai]02LOK@0U7Vd09am]00000006`3oool4000004/0oooo001Y0?ooo`030000003o
ool0oooo01L0oooo00<000000?ooo`3oool00`3oool20000000A09bAc@2DSLd0W97=09B=c@000000
Y8fd0:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd0:B=_@2TSK@0Y8fm0:B=]00000003@2DNK@030000000
U7V/09am]02DNJ`0W7fd09Ai[02LOK@0U7V/09am]02DNJ`0W7fd000001P0oooo1000001?0?ooo`00
JP3oool2000001L0oooo00<000000?ooo`3oool01@3oool20000000W09BAc@2LTLd000000:bA_@2T
SKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0000002DNK@0U7fd09Ai]02D
OK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0000009Ai]02LOK@0U7Vd09am]02D
NK@0W7fd09Ai]02LOK@0000001D0oooo1000001C0?ooo`00K03oool00`000000oooo0?ooo`0D0?oo
o`030000003oool0oooo00L0oooo0P0000003P2LTLd000000:B=_@2TSK@0Y8fm0:B=]02TSKd0Y8fd
0:B=_@2TSK@0Y8fm0:B=]02TSKd000003@2DNK@01`000000W7fd09Ai[02LOK@0U7V/09am]02DNJ`0
0P00000B0?ooo`@00000E`3oool006d0oooo00<000000?ooo`3oool04`3oool00`000000oooo0?oo
o`090?ooo`@0000001h0Y8fm0:bA_@2TSKd0[96m0:B=_@2/TKd0Y8fm0:bA_@2TSKd0000009Ai]02D
OK@0U7Vd09Am]02DNK@0U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai]0000000U7Vd09am]02D
NK@0W7fd09Ai]000000A0?ooo`<00000F`3oool006h0oooo00<000000?ooo`3oool04P3oool00`00
0000oooo0?ooo`0=0?ooo`@0000000H0Y8fm0:B=]02TSKd0Y8fd0:B=_@00000=09Ai]0060000002L
OK@0U7V/09am]02DNJ`000003P3oool4000005h0oooo001_0?ooo`8000004P3oool00`000000oooo
0?ooo`0A0?ooo`@000000180Y8fm0:bA_@000000U7fd09Ai]02DOK@0U7Vd09Am]02DNK@0U7fd09Ai
]02DOK@0U7Vd09Am]02DNK@0U7fd0000002LOK@2000000/0oooo1000001R0?ooo`00L@3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo01D0oooo2@00000709Ai]08000002@3oool40000
06H0oooo001b0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool07`3oool8000000H0
oooo1000001Z0?ooo`00L`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo02T0oooo
1000001^0?ooo`00M03oool2000000`0oooo00<000000?ooo`3oool09P3oool300000780oooo001f
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool08P3oool4000007D0oooo001g0?oo
o`030000003oool0oooo00P0oooo00<000000?ooo`3oool07P3oool4000007T0oooo001h0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool06P3oool4000007d0oooo001i0?ooo`030000
003oool0oooo00H0oooo00<000000?ooo`3oool05P3oool400000840oooo001j0?ooo`8000001P3o
ool00`000000oooo0?ooo`0C0?ooo`<00000Q@3oool007`0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0?0?ooo`@00000R03oool007d0oooo00<000000?ooo`3oool00P3oool00`00
0000oooo0?ooo`0;0?ooo`@00000S03oool007h0oooo00D000000?ooo`3oool0oooo000000090?oo
o`@00000T03oool007l0oooo0P0000000`3oool000000?ooo`040?ooo`@00000U03oool00840oooo
0P0000000`3oool0000000000002000009P0oooo00220?ooo`800000W03oool00?l0oooo8@3oool0
0?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool0
0?l0oooo8@3oool00?l0oooo8@3oool00?l0oooo8@3oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {232.562, 0}} -> {-0.121189, -0.0181932, \
0.003842, 0.003842}}]
}, Open  ]],

Cell["\<\
Note that the values need to be given in a matrix corresponding to \
the outer product of the grid coordinates. \
\>", "Text",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["\<\
NDSolve`FiniteDifferenceDerivative does not compute weights for \
sums of derivatives. This means that for common operators like the Laplacian, \
you need to combine two approximations.\
\>", "Text",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["\<\
This makes a function that approximates the Laplacian operator on a \
tensor product grid.\
\>", "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \(lap[values_, \ {xgrid_, \ ygrid_}]\  := \ 
      NDSolve`FiniteDifferenceDerivative[{2, 0}, {xgrid, \ ygrid}, \ 
          values] + \ 
        NDSolve`FiniteDifferenceDerivative[{0, 2}, {xgrid, \ ygrid}, \ 
          values]\)], "Input",
  CellLabel->"In[26]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["\<\
This uses the function to approximate the Laplacian for the same \
grid and Gaussian function used in the previous example.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(ListPlot3D[lap[values, \ {xgrid, \ ygrid}], \ 
        MeshRange \[Rule] {{0, 8}, {0, 10}}];\)\)], "Input",
  CellLabel->"In[27]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
3.1984e-017 1.04977 -0.0679587 1.04977 [
[.02757 .24618 -6 -8.70276 ]
[.02757 .24618 0 .29724 ]
[.17646 .18758 -5.89944 -9 ]
[.17646 .18758 .10056 0 ]
[.3345 .12542 -5.5867 -9 ]
[.3345 .12542 .4133 0 ]
[.50253 .05935 -5.27396 -9 ]
[.50253 .05935 .72604 0 ]
[.68156 -0.01098 -4.96123 -9 ]
[.68156 -0.01098 1.03877 0 ]
[.70096 -0.00478 0 -6.26206 ]
[.70096 -0.00478 6 2.73794 ]
[.76745 .09703 0 -6.16187 ]
[.76745 .09703 6 2.83813 ]
[.82812 .18993 0 -6.07246 ]
[.82812 .18993 6 2.92754 ]
[.88369 .27502 0 -5.99218 ]
[.88369 .27502 6 3.00782 ]
[.93478 .35327 0 -5.9197 ]
[.93478 .35327 6 3.0803 ]
[.98191 .42546 0 -5.85393 ]
[.98191 .42546 12 3.14607 ]
[.02429 .26403 -24 -2.74141 ]
[.02429 .26403 0 6.25859 ]
[.01689 .31038 -24 -2.78059 ]
[.01689 .31038 0 6.21941 ]
[.00927 .35815 -24 -2.82108 ]
[.00927 .35815 0 6.17892 ]
[.00141 .4074 -24 -2.86297 ]
[.00141 .4074 0 6.13703 ]
[-0.0067 .45821 -6 -2.90632 ]
[-0.0067 .45821 0 6.09368 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.18558 .19702 m
.19014 .20174 L
s
[(2)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(4)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(6)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(8)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.07346 .24092 m
.0763 .24365 L
s
.11029 .2265 m
.1131 .22926 L
s
.14766 .21187 m
.15043 .21466 L
s
.22407 .18195 m
.22677 .18482 L
s
.26314 .16665 m
.2658 .16956 L
s
.3028 .15112 m
.30541 .15406 L
s
.38395 .11935 m
.38647 .12237 L
s
.42547 .10309 m
.42794 .10615 L
s
.46763 .08658 m
.47006 .08968 L
s
.55397 .05277 m
.55629 .05595 L
s
.59817 .03546 m
.60044 .03868 L
s
.64309 .01787 m
.6453 .02113 L
s
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.68874 0 m
.68263 .00239 L
s
[(0)] .70096 -0.00478 -1 .39157 Mshowa
.75514 .10158 m
.74899 .10385 L
s
[(2)] .76745 .09703 -1 .3693 Mshowa
.81573 .19425 m
.80953 .19642 L
s
[(4)] .82812 .18993 -1 .34944 Mshowa
.87123 .27915 m
.865 .28122 L
s
[(6)] .88369 .27502 -1 .3316 Mshowa
.92227 .35722 m
.91601 .35919 L
s
[(8)] .93478 .35327 -1 .31549 Mshowa
.96935 .42924 m
.96306 .43113 L
s
[(10)] .98191 .42546 -1 .30087 Mshowa
.125 Mabswid
.70593 .0263 m
.70226 .02771 L
s
.72272 .05198 m
.71904 .05338 L
s
.73912 .07706 m
.73543 .07845 L
s
.7708 .12553 m
.7671 .12688 L
s
.78611 .14895 m
.78241 .15028 L
s
.80108 .17185 m
.79737 .17317 L
s
.83006 .21617 m
.82633 .21745 L
s
.84407 .23761 m
.84035 .23888 L
s
.8578 .2586 m
.85407 .25986 L
s
.88439 .29928 m
.88065 .3005 L
s
.89727 .31899 m
.89353 .3202 L
s
.9099 .3383 m
.90615 .3395 L
s
.93439 .37576 m
.93063 .37693 L
s
.94627 .39394 m
.94251 .3951 L
s
.95792 .41176 m
.95416 .41291 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.03651 .25925 m
.04262 .25687 L
s
[(-0.4)] .02429 .26403 1 -0.3908 Mshowa
.02915 .3057 m
.03528 .30336 L
s
[(-0.3)] .01689 .31038 1 -0.38209 Mshowa
.02156 .35356 m
.02771 .35127 L
s
[(-0.2)] .00927 .35815 1 -0.37309 Mshowa
.01374 .40291 m
.01991 .40067 L
s
[(-0.1)] .00141 .4074 1 -0.36378 Mshowa
.00567 .45383 m
.01186 .45163 L
s
[(0)] -0.0067 .45821 1 -0.35415 Mshowa
.125 Mabswid
.03505 .26843 m
.03872 .267 L
s
.03359 .27766 m
.03726 .27624 L
s
.03212 .28695 m
.03579 .28554 L
s
.03064 .2963 m
.03431 .29489 L
s
.02765 .31515 m
.03133 .31376 L
s
.02614 .32467 m
.02982 .32328 L
s
.02462 .33424 m
.02831 .33285 L
s
.0231 .34387 m
.02678 .34249 L
s
.02002 .36331 m
.02371 .36194 L
s
.01846 .37312 m
.02216 .37176 L
s
.0169 .38299 m
.02059 .38163 L
s
.01533 .39292 m
.01902 .39157 L
s
.01215 .41297 m
.01585 .41163 L
s
.01054 .42309 m
.01425 .42175 L
s
.00893 .43327 m
.01264 .43194 L
s
.00731 .44352 m
.01102 .44219 L
s
.00403 .4642 m
.00774 .46289 L
s
.00237 .47464 m
.00609 .47334 L
s
.00071 .48515 m
.00443 .48386 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
.427 .696 .976 r
.35765 .75149 .39885 .7832 .45292 .77909 .41226 .75454 Metetra
.62 .814 .975 r
.41226 .75454 .45292 .77909 .50846 .76738 .46887 .74012 Metetra
.709 .82 .934 r
.46887 .74012 .50846 .76738 .56519 .75108 .52671 .715 Metetra
.738 .781 .888 r
.52671 .715 .56519 .75108 .62297 .73118 .58537 .68359 Metetra
.693 .722 .872 r
.58537 .68359 .62297 .73118 .68191 .71295 .64504 .65944 Metetra
.607 .669 .883 r
.64504 .65944 .68191 .71295 .74234 .69949 .70655 .65016 Metetra
.546 .663 .913 r
.70655 .65016 .74234 .69949 .80433 .68778 .77024 .64849 Metetra
.572 .73 .949 r
.77024 .64849 .80433 .68778 .86751 .67242 .8354 .64064 Metetra
.643 .808 .962 r
.8354 .64064 .86751 .67242 .9316 .65222 .90123 .62174 Metetra
.76 .857 .922 r
.90123 .62174 .9316 .65222 .99594 .62381 .96617 .58417 Metetra
.388 .657 .965 r
.31461 .71471 .35765 .75149 .41226 .75454 .36969 .72122 Metetra
.725 .792 .905 r
.36969 .72122 .41226 .75454 .46887 .74012 .42765 .69279 Metetra
.785 .743 .814 r
.42765 .69279 .46887 .74012 .52671 .715 .48668 .64291 Metetra
.768 .676 .761 r
.48668 .64291 .52671 .715 .58537 .68359 .54582 .58531 Metetra
.689 .598 .749 r
.54582 .58531 .58537 .68359 .64504 .65944 .60534 .54909 Metetra
.567 .515 .757 r
.60534 .54909 .64504 .65944 .70655 .65016 .66707 .54946 Metetra
.461 .471 .782 r
.66707 .54946 .70655 .65016 .77024 .64849 .73223 .57211 Metetra
.464 .532 .842 r
.73223 .57211 .77024 .64849 .8354 .64064 .79987 .58773 Metetra
.57 .676 .911 r
.79987 .58773 .8354 .64064 .90123 .62174 .86825 .58162 Metetra
.774 .819 .888 r
.86825 .58162 .90123 .62174 .96617 .58417 .9343 .53862 Metetra
.369 .641 .96 r
.26956 .67499 .31461 .71471 .36969 .72122 .3251 .68263 Metetra
.782 .779 .85 r
.3251 .68263 .36969 .72122 .42765 .69279 .3848 .63809 Metetra
.826 .731 .761 r
.3848 .63809 .42765 .69279 .48668 .64291 .44559 .56204 Metetra
.802 .675 .724 r
.44559 .56204 .48668 .64291 .54582 .58531 .5057 .47837 Metetra
.709 .598 .73 r
.5057 .47837 .54582 .58531 .60534 .54909 .5652 .43049 Metetra
.525 .48 .749 r
.5652 .43049 .60534 .54909 .66707 .54946 .62674 .43995 Metetra
.321 .372 .758 r
.62674 .43995 .66707 .54946 .73223 .57211 .69266 .48658 Metetra
.295 .412 .812 r
.69266 .48658 .73223 .57211 .79987 .58773 .76226 .52684 Metetra
.464 .598 .901 r
.76226 .52684 .79987 .58773 .86825 .58162 .83304 .53551 Metetra
.776 .802 .874 r
.83304 .53551 .86825 .58162 .9343 .53862 .90054 .48938 Metetra
.356 .641 .962 r
.22217 .63449 .26956 .67499 .3251 .68263 .27824 .64184 Metetra
.801 .778 .832 r
.27824 .64184 .3251 .68263 .3848 .63809 .3397 .58914 Metetra
.853 .753 .752 r
.3397 .58914 .3848 .63809 .44559 .56204 .4026 .50058 Metetra
.847 .725 .73 r
.4026 .50058 .44559 .56204 .5057 .47837 .46449 .40482 Metetra
.76 .68 .773 r
.46449 .40482 .5057 .47837 .5652 .43049 .52489 .35133 Metetra
.457 .523 .838 r
.52489 .35133 .5652 .43049 .62674 .43995 .58664 .36411 Metetra
.029 .271 .769 r
.58664 .36411 .62674 .43995 .69266 .48658 .65286 .42066 Metetra
.037 .318 .804 r
.65286 .42066 .69266 .48658 .76226 .52684 .72345 .47154 Metetra
.368 .581 .924 r
.72345 .47154 .76226 .52684 .83304 .53551 .79583 .48657 Metetra
.777 .801 .872 r
.79583 .48657 .83304 .53551 .90054 .48938 .86504 .43865 Metetra
.342 .648 .967 r
.17209 .59453 .22217 .63449 .27824 .64184 .22874 .60106 Metetra
.785 .787 .854 r
.22874 .60106 .27824 .64184 .3397 .58914 .29131 .55388 Metetra
.864 .803 .788 r
.29131 .55388 .3397 .58914 .4026 .50058 .35592 .47435 Metetra
.893 .834 .781 r
.35592 .47435 .4026 .50058 .46449 .40482 .42 .38708 Metetra
.826 .91 .896 r
.42 .38708 .46449 .40482 .52489 .35133 .48245 .33649 Metetra
.051 .59 .851 r
.48245 .33649 .52489 .35133 .58664 .36411 .54555 .34448 Metetra
.489 0 0 r
.54555 .34448 .58664 .36411 .65286 .42066 .6124 .39047 Metetra
0 .279 .71 r
.6124 .39047 .65286 .42066 .72345 .47154 .68343 .42999 Metetra
.356 .659 .97 r
.68343 .42999 .72345 .47154 .79583 .48657 .75663 .43705 Metetra
.779 .811 .879 r
.75663 .43705 .79583 .48657 .86504 .43865 .82775 .38776 Metetra
.363 .665 .972 r
.11913 .55362 .17209 .59453 .22874 .60106 .17653 .5578 Metetra
.724 .782 .898 r
.17653 .5578 .22874 .60106 .29131 .55388 .23931 .52493 Metetra
.824 .849 .866 r
.23931 .52493 .29131 .55388 .35592 .47435 .3046 .46932 Metetra
.865 .948 .867 r
.3046 .46932 .35592 .47435 .42 .38708 .37044 .40556 Metetra
.551 .923 .776 r
.37044 .40556 .42 .38708 .48245 .33649 .43552 .36427 Metetra
.234 0 0 r
.43552 .36427 .48245 .33649 .54555 .34448 .50108 .36128 Metetra
.437 0 0 r
.50108 .36128 .54555 .34448 .6124 .39047 .56943 .38161 Metetra
0 .477 .713 r
.56943 .38161 .6124 .39047 .68343 .42999 .64115 .39447 Metetra
.5 .781 .995 r
.64115 .39447 .68343 .42999 .75663 .43705 .71503 .38428 Metetra
.772 .809 .882 r
.71503 .38428 .75663 .43705 .82775 .38776 .78828 .33503 Metetra
.437 .693 .973 r
.06335 .50918 .11913 .55362 .17653 .5578 .12181 .50836 Metetra
.639 .748 .928 r
.12181 .50836 .17653 .5578 .23931 .52493 .1844 .48837 Metetra
.712 .826 .935 r
.1844 .48837 .23931 .52493 .3046 .46932 .24953 .45659 Metetra
.733 .938 .953 r
.24953 .45659 .3046 .46932 .37044 .40556 .3163 .41782 Metetra
.453 .883 .814 r
.3163 .41782 .37044 .40556 .43552 .36427 .38371 .38677 Metetra
.38371 .38677 .43552 .36427 .50108 .36128 .4521 .37163 Metetra
.4521 .37163 .50108 .36128 .56943 .38161 .52264 .36453 Metetra
.419 .831 .972 r
.52264 .36453 .56943 .38161 .64115 .39447 .59567 .35052 Metetra
.647 .814 .964 r
.59567 .35052 .64115 .39447 .71503 .38428 .67063 .32429 Metetra
.75 .787 .883 r
.67063 .32429 .71503 .38428 .78828 .33503 .74617 .27763 Metetra
.55 .72 .952 r
.00495 .4584 .06335 .50918 .12181 .50836 .06515 .44814 Metetra
.624 .712 .909 r
.06515 .44814 .12181 .50836 .1844 .48837 .12834 .42918 Metetra
.655 .73 .904 r
.12834 .42918 .1844 .48837 .24953 .45659 .19377 .40486 Metetra
.679 .771 .92 r
.19377 .40486 .24953 .45659 .3163 .41782 .26114 .37628 Metetra
.663 .816 .957 r
.26114 .37628 .3163 .41782 .38371 .38677 .32991 .34932 Metetra
.611 .825 .982 r
.32991 .34932 .38371 .38677 .4521 .37163 .40016 .32739 Metetra
.6 .796 .974 r
.40016 .32739 .4521 .37163 .52264 .36453 .47239 .30714 Metetra
.637 .759 .936 r
.47239 .30714 .52264 .36453 .59567 .35052 .54681 .28252 Metetra
.665 .735 .902 r
.54681 .28252 .59567 .35052 .67063 .32429 .62321 .25221 Metetra
.705 .743 .88 r
.62321 .25221 .67063 .32429 .74617 .27763 .70107 .21247 Metetra
0 g
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0)] .02757 .24618 1 .93395 Mshowa
.18558 .19702 m
.19014 .20174 L
s
[(2)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(4)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(6)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(8)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.07346 .24092 m
.0763 .24365 L
s
.11029 .2265 m
.1131 .22926 L
s
.14766 .21187 m
.15043 .21466 L
s
.22407 .18195 m
.22677 .18482 L
s
.26314 .16665 m
.2658 .16956 L
s
.3028 .15112 m
.30541 .15406 L
s
.38395 .11935 m
.38647 .12237 L
s
.42547 .10309 m
.42794 .10615 L
s
.46763 .08658 m
.47006 .08968 L
s
.55397 .05277 m
.55629 .05595 L
s
.59817 .03546 m
.60044 .03868 L
s
.64309 .01787 m
.6453 .02113 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[27]:=",
  ImageSize->{288, 233.562},
  ImageMargins->{{1, 0}, {0, 1}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jB000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00``3oool2000005/0oooo00320?ooo`040000003oool0oooo000005X0
oooo00320?ooo`040000003oool0oooo000005X0oooo00330?ooo`800000F`3oool00<80oooo00@0
00000?ooo`3oool00000FP3oool00<<0oooo0P00001K0?ooo`00c@3oool200000540oooo003<0?oo
o`040000003oool0oooo00000500oooo003<0?ooo`040000003oool0oooo00000500oooo003<0?oo
o`040000003oool0oooo00000500oooo003<0?ooo`040000003oool0oooo00000500oooo00360?oo
o`8000001@3oool200000540oooo00340?ooo`D00000E`3oool00<40oooo0`0000030?ooo`800000
E`3oool00;l0oooo0P0000060?ooo`030000003oool0000005H0oooo002l0?ooo`<00000203oool0
10000000oooo0?ooo`00001E0?ooo`00^P3oool3000000X0oooo00@000000?ooo`3oool00000E@3o
ool00;L0oooo0`0000020?ooo`030000003oool0oooo00P0oooo00D000000?ooo`3oool0oooo0000
001D0?ooo`00T`3oool3000001l0oooo0P00000@0?ooo`040000003oool0oooo0?ooo`800000D`3o
ool009<0oooo00@000000?ooo`3oool000006`3oool3000001<0oooo00D000000?ooo`3oool0oooo
0000001C0?ooo`00T`3oool010000000oooo0?ooo`00000I0?ooo`8000005P3oool00`000000oooo
0?ooo`020?ooo`030000003oool0oooo0500oooo002C0?ooo`<000005`3oool3000000030?ooo`00
0000oooo01D0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`1?0?ooo`00U03oool0
0`000000oooo0?ooo`0D0?ooo`8000006`3oool00`000000oooo0?ooo`030?ooo`030000003oool0
oooo04l0oooo002D0?ooo`<000004@3oool3000001d0oooo00<000000?ooo`3oool0103oool00`00
0000oooo0?ooo`1>0?ooo`00YP3oool200000200oooo00<000000?ooo`3oool01@3oool00`000000
oooo0?ooo`1=0?ooo`00X`3oool300000280oooo00<000000?ooo`3oool0103oool2000004l0oooo
002Q0?ooo`8000000`3oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo00H0oooo00<0
00000?ooo`3oool0C03oool009h0oooo0`00000W0?ooo`030000003oool0oooo00H0oooo00<00000
0?ooo`3oool0C03oool009/0oooo0`00000Z0?ooo`030000003oool0oooo00L0oooo00<000000?oo
o`3oool0B`3oool009T0oooo0P00000]0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3o
ool0BP3oool009H0oooo0`0000000`3oool000000?ooo`0/0?ooo`030000003oool0oooo00P0oooo
00<000000?ooo`3oool0BP3oool009@0oooo0P0000050?ooo`030000003oool0oooo02X0oooo00<0
00000?ooo`3oool0203oool2000004/0oooo002A0?ooo`<00000=03oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0oooo00D0oooo1000000o0?ooo`00S`3oool2000003L0oooo00<000000?oo
o`3oool02P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo0400oooo002<0?ooo`@0
0000>03oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3o
ool0?`3oool006X0oooo0`00000M0?ooo`800000?@3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo00D0oooo00<000000?ooo`3oool0?P3oool006/0oooo00<000000?ooo`3oool06@3o
ool3000003l0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`030?ooo`040000003o
ool0oooo000003l0oooo001X0?ooo`D00000603oool200000480oooo00<000000?ooo`3oool0303o
ool00`000000oooo0?ooo`030?ooo`800000@03oool006P0oooo00@000000?ooo`3oool000005P3o
ool4000004<0oooo00<000000?ooo`3oool02`3oool00`000000oooo000000160?ooo`00J@3oool0
0`000000oooo0000000D0?ooo`800000A`3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo04@0oooo001Z0?ooo`8000004@3oool3000004T0oooo00<000000?ooo`3oool03P3oool00`00
0000oooo0?ooo`130?ooo`00N`3oool2000004`0oooo00<000000?ooo`3oool03`3oool00`000000
oooo0?ooo`120?ooo`00N03oool3000004h0oooo00<000000?ooo`3oool03`3oool00`000000oooo
0?ooo`120?ooo`00MP3oool200000080oooo00<000000?ooo`3oool0C03oool00`000000oooo0?oo
o`0@0?ooo`030000003oool0oooo0440oooo001c0?ooo`<00000D`3oool00`000000oooo0?ooo`0@
0?ooo`800000@P3oool00740oooo0P00001F0?ooo`030000003oool0oooo0140oooo00<000000?oo
o`3oool0@03oool006h0oooo0`00001H0?ooo`030000003oool0oooo0180oooo00<000000?ooo`3o
ool0?`3oool006`0oooo0P0000020?ooo`030000003oool0oooo05H0oooo00<000000?ooo`3oool0
4`3oool00`000000oooo0?ooo`0n0?ooo`00J@3oool3000005d0oooo00<000000?ooo`3oool04`3o
ool00`000000oooo0?ooo`0n0?ooo`00I`3oool200000600oooo00<000000?ooo`3oool0503oool0
0`000000oooo0?ooo`0m0?ooo`00I03oool300000680oooo00<000000?ooo`3oool04`3oool00`00
0000oooo0000000n0?ooo`00?`3oool4000001l0oooo0P0000000`3oool000000?ooo`1S0?ooo`03
0000003oool0oooo01@0oooo00<000000?ooo`3oool0?03oool003l0oooo00<000000?ooo`3oool0
7@3oool3000006P0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`0k0?ooo`00@03o
ool00`000000oooo0?ooo`0J0?ooo`800000J`3oool00`000000oooo0?ooo`0E0?ooo`030000003o
ool0oooo03/0oooo00110?ooo`030000003oool0oooo01H0oooo0`00001]0?ooo`030000003oool0
oooo01H0oooo00<000000?ooo`3oool0>P3oool003l0oooo00@000000?ooo`3oool000005@3oool2
000000030?ooo`000000oooo06d0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`0i
0?ooo`00@03oool2000001<0oooo0`00001b0?ooo`030000003oool0oooo01H0oooo0P0000090?oo
o`<00000;`3oool005<0oooo0P00001e0?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3o
ool01`3oool00`000000oooo0?ooo`0^0?ooo`00D03oool3000007L0oooo00<000000?ooo`3oool0
6@3oool00`000000oooo0?ooo`030?ooo`D00000;`3oool004h0oooo0P0000000`3oool000000?oo
o`1g0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool00`3oool010000000oooo0?oo
o`00000`0?ooo`00B`3oool3000007`0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?oo
o`030?ooo`030000003oool000000300oooo00180?ooo`<00000O`3oool00`000000oooo0?ooo`0K
0?ooo`030000003oool0oooo00<0oooo0P00000`0?ooo`00AP3oool200000880oooo00<000000?oo
o`3oool06@3oool3000003L0oooo00130?ooo`<000000P3oool00`000000oooo0?ooo`1o0?ooo`03
0000003oool0oooo01`0oooo00<000000?ooo`3oool0=03oool00440oooo0P0000270?ooo`030000
003oool0oooo01d0oooo00<000000?ooo`3oool0<`3oool003h0oooo0`0000290?ooo`030000003o
ool0oooo01d0oooo00<000000?ooo`3oool0<`3oool003`0oooo0`00002;0?ooo`8000007`3oool0
0`000000oooo0?ooo`0b0?ooo`006P3oool2000001d0oooo0`00002=0?ooo`@000007P3oool20000
03<0oooo000I0?ooo`040000003oool0oooo000001X0oooo0P00002>0?ooo`80000000@0];kV0;Bn
gP00000000007`3oool00`000000oooo0?ooo`0a0?ooo`006@3oool010000000oooo0?ooo`00000F
0?ooo`@00000SP3oool2000000070;BngP2d_^H0];kN0;BniP000000];kV0000000O0?ooo`030000
003oool0oooo0300oooo000I0?ooo`040000003oool0oooo000001D0oooo0`00002>0?ooo`800000
00X0];kV0;BngP2d_^H0];kN0;BniP2d_]h000000;BngP2d_^H000007`3oool00`000000oooo0?oo
o`0_0?ooo`006@3oool010000000oooo0?ooo`00000B0?ooo`<00000S`3oool20000000<0;BngP2d
_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H000000;BniP2d_]h000007`3oool00`000000oooo0?oo
o`0_0?ooo`006P3oool200000140oooo0P00002@0?ooo`80000000l0];kV0;BngP2d_^H0];kN0;Bn
iP2d_]h0];kV0;BngP2d_^H0];kN0000002d_]h0];kV0;BngP0000007P3oool200000300oooo000Z
0?ooo`<00000T03oool20000000B0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;Bn
iP2d_]h0];kV0000002d_^H0];kN0;BniP2d_]h000007`3oool00`000000oooo0?ooo`0]0?ooo`00
:03oool2000000030?ooo`000000oooo08h0oooo0P0000003@2d_^H0];kN0;BniP2d_]h0];kV0;Bn
gP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H00P0000001@2d_]h0];kV0;BngP2d_^H0000001l0
oooo00<000000?ooo`3oool0;@3oool002D0oooo0`00002A0?ooo`80000000`0];kN0;BniP2d_]h0
];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H3000000080;BniP2d_]h000000;BngP2d
_^H0];kN0;BniP00000O0?ooo`030000003oool0oooo02`0oooo000S0?ooo`800000T@3oool30000
000;0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H00`000000302d_^H0
];kN0;BniP2d_]h0];kV0000002d_^H0];kN0;BniP2d_]h0];kV000001l0oooo00<000000?ooo`3o
ool0:`3oool00240oooo0P00002A0?ooo`80000000`0[;kV0:bjiP000000];kN0;BniP2d_]h0];kV
0;BngP2d_^H0];kN0;BniP2d_]h20000000@0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0
];kV0000002d_^H0];kN0;BniP2d_]h0];kV000001h0oooo0P00000/0?ooo`00803oool3000008h0
oooo0`0000002`2T^^H0[;kV0:BjiP2/_^H000000;BniP2d_]h0];kV0;BngP2d_^H0];kN00<00000
0180];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H000000;BniP2d
_]h0];kV0;BngP00000N0?ooo`8000001P3oool3000002<0oooo000=0?ooo`800000103oool20000
00D0oooo0`0000040?ooo`030000003oool0000008/0oooo0P0000002`2/^^H0[;kV0:bjiP2/_^H0
[;[V0:bniP2/^^H0[;kV0000002d_^H0];kN00<0000001H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0
];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0000002d_^H0];kN0;BniP2d_]h0
00007`3oool00`000000oooo0?ooo`030?ooo`040000003oool0oooo00000280oooo000<0?ooo`04
0000003oool0oooo000000/0oooo00<000000?ooo`3oool00P3oool200000080oooo00<000000?oo
o`3oool0Q@3oool30000000:0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP<0
000001X0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;Bn
iP2d_]h0];kV0;BngP2d_^H0];kN0000002d_]h0];kV0;BngP2d_^H0];kN000001h0oooo00<00000
0?ooo`3oool00`3oool010000000oooo0?ooo`00000R0?ooo`00303oool010000000oooo0?ooo`00
00080?ooo`D000000`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0880oooo0P00
00002P2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H30000000M0:bjiP000000
];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0
];kV0;BngP2d_^H0];kN0;BniP2d_]h000000;BngP2d_^H0];kN0;BniP0000007`3oool00`000000
oooo0?ooo`020?ooo`<000008`3oool000H0oooo100000020?ooo`040000003oool0oooo000000P0
oooo00@000000?ooo`3oool00000103oool2000000@0oooo00<000000?ooo`3oool0OP3oool30000
00090:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V00<000000240Y;[V0:bniP2T^^H0
[;kV0:BjiP000000];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0
];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h000000;BngP2d_^H0];kN0;BniP000000
7P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0280oooo000<0?ooo`040000003o
ool0oooo000000T0oooo00<000000?ooo`000000103oool00`000000oooo0?ooo`040?ooo`030000
003oool0oooo07/0oooo0P0000002@2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP03
0000000Q0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0000002d_]h0];kV0;BngP2d
_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d
_]h0];kV0;BngP000000];kN0080000000<0aL[V0000003oool0703oool2000000@0oooo0`00000R
0?ooo`003@3oool2000000/0oooo0P0000040?ooo`030000003oool0oooo00D0oooo0P00001h0?oo
o`<0000000T0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H00P0000008P2T^^H0[;kV
0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP000000];kV0;BngP2d_^H0];kN
0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV
0;BngP2d_^H2000000040;g6gP2mb^H0_LKN000001l0oooo00<000000?ooo`3oool09P3oool001l0
oooo0P0000090?ooo`030000003oool0oooo0780oooo0`0000002@2T`_H0Y<;^0000002/^^H0[;kV
0:bjiP2/_^H0[;[V0:bniP030000000S0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV
0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0000002d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN
0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H00`0000001@35b^H0_L[N
0<G:iP2mb]h0000001h0oooo00<000000?ooo`3oool09P3oool001l0oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`1^0?ooo`<0000000T0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h00000
0:BjiP2/_^H00`000000902/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV
0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H000000;BniP2d_]h0];kV0;BngP2d_^H0];kN
0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP80000000T0_L[V0;g6gP2mb^H0
00000;g:iP2ma]h0_L[V0;g6gP0000007P3oool00`000000oooo0?ooo`0U0?ooo`007`3oool20000
00/0oooo00<000000?ooo`3oool0J@3oool4000000090:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2
mP2T`^h0Y<;f00<0000003<0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bn
iP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0000002d_]h0];kV0;Bn
gP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP000000_L[N0<G:
iP2mb]h0aL[V0;g:gP000000_L[N0<G:iP2mb]h0aL[V0;g:gP0000007P3oool00`000000oooo0?oo
o`0T0?ooo`007`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo06D0oooo0`000000
2P2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h30000000W0:C2kP000000[;kV
0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V
0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H000000;BngP2d_^H0];kN0;BniP2d_]h0];kV
0;BngP2d_^H0];kN0;BniP2d_]h0];kV0080000000d0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN
0000002ma]h0_L[V0;g6gP2mb^H0000001h0oooo0P00000U0?ooo`007`3oool2000000d0oooo00<0
00000?ooo`3oool0H@3oool30000000:0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^
0:C2mP<0000003X0Y<;f0:C2kP2T`_H0Y<;^0:C2mP000000[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0
[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0
[;[V0:bniP2/^^H000000;BngP2d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0;BniP000000
aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0000002mb]h0aL[V0;g:gP35b^H0
00007P3oool00`000000oooo0?ooo`0S0?ooo`007`3oool00`000000oooo0?ooo`0=0?ooo`030000
003oool0oooo05d0oooo0`0000002P2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T
`^h30000000Z0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^0000002/_^H0Y;[V0:bn
iP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:Bj
iP2/_^H0Y;[V0:bniP2T^^H0[;kV0000002d_^H0];kN0;BniP2d_]h0];kV0;BngP2d_^H0];kN0P00
00004P2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP000000_LKN
0;g:iP2ma]h0_L[V000001h0oooo00<000000?ooo`3oool08P3oool000/0oooo0P0000040?ooo`80
0000103oool2000000H0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`1H0?ooo`@0
000000/0W<[f09c>o`000000Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP020000000[0:C2
mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0000002/^^H0[;kV0:bj
iP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bn
iP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP000000];kV0;BngP2d_^H0];kN0;BniP020000000E0;g:
gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H000000<G:
iP2mb]h0aL[V0;g:gP35b^H0000001d0oooo00<000000?ooo`3oool08P3oool000X0oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`0000040?ooo`8000004@3oool00`000000oooo
0?ooo`1D0?ooo`<0000000`0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0W<[o0000002T`^h0W<;^0:C2
kP2L`^h30000001409c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^
09c2kP2T`^h0W<;^0000002T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V
0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP000000];kV
0;BngP2d_^H000000;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN
0;g:iP2ma]h0_L[V0;g6gP2mb^H000000;g:iP2ma]h0_L[V0;g6gP00000N0?ooo`030000003oool0
oooo0240oooo000:0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0A0?ooo`030000003oool0oooo04l0oooo100000002`2Lc_l0W<[f09c>o`2L
b_H0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H0W<ko00@0000002d0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T
`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0000002/
^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/
_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0000002d_]h00P0000006@35b^H0_L[N0<G:iP2m
b]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35
b^H000000<G:iP2mb]h0aL[V0;g:gP0000007@3oool200000280oooo00040?ooo`@000000P3oool0
10000000oooo0?ooo`00000;0?ooo`030000003oool0oooo0080oooo0P00000C0?ooo`030000003o
ool0oooo04X0oooo100000003@2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0W<[o09C:mP2Lb_l0
U<[f09c:o`2Db_H00`000000;P2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0
W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0000002T^^H0
[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0
Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H20000000L0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2m
b^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0000002m
a]h0_L[V0;g6gP2mb^H0_LKN000001d0oooo00<000000?ooo`3oool0803oool000X0oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo01<0oooo
00<000000?ooo`3oool0AP3oool30000000>09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H0
W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H30000000^09c:mP2Lc_l0W<[f0000002T`^h0Y<;f0:C2kP2T
`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T
`^h0Y<;f0:C2kP2T`_H000000:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/
_^H0[;[V0:bniP2/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP<0000001l0Y=;f0:C>mP000000_L[N0<G:
iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:
gP35b^H0_L[N0<G:iP2mb]h000000;g:gP35b^H0_L[N0<G:iP2mb]h0000001d0oooo00<000000?oo
o`3oool07`3oool000/0oooo0P00000:0?ooo`8000001@3oool00`000000oooo0?ooo`0D0?ooo`03
0000003oool0oooo0480oooo0`0000003P2Ld_l0W=;f0000002Db_H0W<[o09C:mP2Lb_l0U<[f09c:
o`2Db_H0W<[o09C:mP2Lb_l0U<[f0`000000;`2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H00000
09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^
0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP000000[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV
0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP2/_^H0Y;[V008000000280Y=;f0:C>mP2Td_H0Y<kf
0:CBmP000000_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V
0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0000002ma]h0_L[V0;g6gP2mb^H00000
7P3oool00`000000oooo0?ooo`0N0?ooo`007@3oool2000001L0oooo00<000000?ooo`3oool0?P3o
ool30000000>09cFo`2Ld_l0W=Ko09cBo`2Le_l0W=;o0000002Lb_H0W<ko09c:mP2Lc_l0W<[f09c>
o`2Lb_H30000000_09c:mP2Lc_l0W<[f09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H00000
0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^
0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP000000[;kV0:bjiP2/_^H0[;[V0:bniP2/^^H0[;kV
0:bjiP2/_^H0[;[V0:bniP2/^^H00`0000009@2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f
00000035b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N
0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h000000;g:gP35b^H0_L[N0<G:iP0000007@3o
ool00`000000oooo0?ooo`0N0?ooo`007@3oool00`000000oooo0?ooo`0G0?ooo`030000003oool0
oooo03X0oooo0`0000002P2Ld_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f09cBo`2Ld_H20000
000309c:o`2Db_H0W<[o0080000002l0U<[f09c:o`2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0
W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0000009c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0
W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP000000
[;kV0:BjiP2/_^H0Y;[V0:bniP2T^^H0[;kV0:BjiP030000000U0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0
Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0000002mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0
_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP000000
_L[V0;g6gP02000000030<G>iP000000oooo01/0oooo0P0000070?ooo`8000005P3oool001d0oooo
0P00000I0?ooo`800000=`3oool30000000?09cFo`2Ld_l0W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0
W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0W=Ko00<0000002l0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H0
W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H000000:C2kP2T`_H0
Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0
Y<;f0:C2kP2T`_H0Y<;^0000002/^^H0[;kV0:bjiP2/_^H0[;[V0:bniP020000000V0:C>mP2Td_H0
Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0000002mb]h0
aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0
_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H2000000040<G>gP35d^H0aLkN000001h0oooo00<000000?oo
o`3oool00`3oool010000000oooo0?ooo`00000E0?ooo`007@3oool00`000000oooo0?ooo`0J0?oo
o`030000003oool0oooo0340oooo0`0000003`2Ld_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f
09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f09cBo`030000000_09cBo`000000W<[o09C:mP2Lb_l0U<[f
09c:o`2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0W<[o
09C:mP000000W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^
0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h000000:BjiP2/_^H00`0000009`2Td_H0Y<kf
0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f
0:C>mP000000_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN
0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP03000000050<G>iP35c]h0aLkV0<G>gP0000007@3o
ool00`000000oooo0?ooo`030?ooo`040000003oool0oooo000001D0oooo000M0?ooo`800000703o
ool00`000000oooo0?ooo`0]0?ooo`<0000000l0W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0W=Ko09cB
o`2Le_l0W=;o09cFo`2Ld_l0W=Ko09cBo`2Le_l00`000000;P2Le_l0W=;o09cFo`2Ld_l0W=Ko0000
002Lc_l0W<[f09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>
o`2Lb_H0W<ko09c:mP2Lc_l0W<[f0000002T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2
kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h40000000c0:CBmP2Tc_H0Y=;f
0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf
0:CBmP2Tc_H0Y=;f0:C>mP000000_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V
0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H000000<GBiP35c]h0aM;V00000035d^H0aLkN
0<GBiP35c]h0000001d0oooo00<000000?ooo`3oool00`3oool2000001H0oooo000M0?ooo`030000
003oool0oooo01`0oooo00<000000?ooo`3oool0:P3oool20000000?0:cBmP2Tc^h000000000002L
d_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f09cBo`2Ld_H0W=;o00<0000002`0W=;o09cBmP2L
d_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0000009c:o`2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`2D
b_H0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`000000Y<;^09c2kP2T
`^h0W<;^0:C2kP2L`^h0Y<;^09c2kP2T`^h0W<;^0:C2kP2L`^h0Y<;^09c2kPD0000002/0J];f06[F
mP1Zd_H000000:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>
mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP000000_L[V0;g6gP2mb^H0_LKN0;g:
iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0080000000/0aLkV0<G>
gP35c^H0aLkN00000035c]h0aLkV0<G>gP35c^H0aLkN0000000M0?ooo`050000003oool0oooo0?oo
o`0000000P3oool00`000000oooo0?ooo`0C0?ooo`002P3oool2000000@0oooo0P0000030?ooo`@0
0000103oool00`000000oooo0?ooo`0M0?ooo`030000003oool0oooo02H0oooo0`0000003P2Tc_H0
[=;f0:C>mP2/d_H0Y<kf0:cBmP000000W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0W=Ko0`0000003@2L
e_l0W=;o09cFo`2Ld_l0W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0W=Ko09cBo`2Le_l00P000000702L
b_H0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>o`2Lb_H0W<ko09c:mP2L
c_l0W<[f09c>o`000000Y<;f0:C2kP2T`_H0Y<;^0:C2mP2T`^h0Y<;f0:C2kP2T`_H40000000m06[B
mP1Ze_l0J];f06[Fo`1Zd_H0J]Ko06[BmP1Ze_l0J];f0000002Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CB
mP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>
mP2Td_H000000<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:iP2mb]h0aL[V0;g:gP35b^H0_L[N0<G:
iP2mb]h000000<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP000000aLkN0<GBiP35c]h0aM;V0000
000L0?ooo`800000103oool2000001H0oooo00090?ooo`040000003oool0oooo000000P0oooo00<0
00000?ooo`3oool0103oool200000200oooo00<000000?ooo`3oool08`3oool20000000:0:C>kP2/
d_H0Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cBmP80000000<0W=;f09cBo`0000000P000000
:@2Ld_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f09cB
o`2Ld_H0W=;o09cBmP000000U<[f09c:o`2Db_H0W<[o09C:mP2Lb_l0U<[f09c:o`2Db_H0W<[o09C:
mP2Lb_l0U<[f09c:o`2Db_H0W<[o09C:mP2Lb_l000000:C2kP2L`^h0Y<;^00D000000480J]Kf06[B
mP1Ze_H0J];f06[FmP1Zd_H0J]Kf06[BmP1Ze_H0J];f06[FmP1Zd_H0J]Kf06[BmP000000Y<kf0:CB
mP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>
mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0000002mb^H0_LKN0;g:iP2ma]h0_L[V0;g6gP2mb^H0_LKN0;g:
iP2ma]h0_L[V0;g6gP000000aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h000000<G>
gP35c^H0aLkN0<G>iP00000M0?ooo`030000003oool0oooo01T0oooo00090?ooo`040000003oool0
oooo000000T0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0P0?ooo`030000003o
ool0oooo01l0oooo0`0000003P2/d_H0Y<kf0:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf0:cBmP2Tc_H0
[=;f0:C>mP2/d_H0Y<kf0P0000008@2Le_l0W=;o09cFo`2Ld_l0W=Ko09cBo`2Le_l0W=;o09cFo`2L
d_l0W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0000009c:mP2L
c_l0W<[f09c>o`2Lb_H0W<ko09c:mP2Lc_l0W<[f09c>o`0;0000000e06[Fo`1Zd_H0J]Ko06[BmP1Z
e_l0J];f06[Fo`1Zd_H0J]Ko06[BmP1Ze_l0J];f06[Fo`1Zd_H0J]Ko06[BmP1Ze_l0J];f06[Fo`1Z
d_H000000:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2T
d_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP000000aL[V0;g:gP35b^H0_L[N0<G:iP2m
b]h0aL[V0;g:gP35b^H00P0000004@35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35
d^H000000<GBiP35c]h0aM;V0<G>gP35d^H0000001d0oooo00<000000?ooo`3oool0603oool000<0
oooo100000020?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3oool20000
0280oooo00<000000?ooo`3oool06`3oool30000000>0:cBmP2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cB
mP2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cBmP2Tc^h30000000G0:C>kP000000W=;f09cBo`2Ld_H0W=;o
09cBmP2Ld_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f
09cBo`2Ld_H030000000203Ej^H0cNKV0=GZiP000000aJk50;f^a@35[/D0_Jk51@000000AP1Zd_H0
J]Kf06[BmP1Ze_H0J];f06[FmP1Zd_H0J]Kf06[BmP1Ze_H0J];f06[FmP1Zd_H0J]Kf06[BmP1Ze_H0
J];f06[FmP1Zd_H000000:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0
Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0000002ma]h0_L[V0;g6gP2mb^H0
_LKN0;g:iP2ma]h0_L[V00000035c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0
aLkV0<G>gP35c^H000000<G>iP35c]h0aLkV0<G>gP00000M0?ooo`030000003oool0oooo01P0oooo
00090?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000`3oool00`000000
oooo0?ooo`0R0?ooo`030000003oool0oooo01P0oooo0P0000003P2/d_H0Y<kf0:cBmP2Tc_H0[=;f
0:C>mP2/d_H0Y<kf0:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf0`0000001@2Tc_H0[=;f0:C>mP2/d_H0
Y<kf0080000000l0W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0W=Ko09cBo`2Le_l0W=;o09cFo`2Ld_l0
W=Ko09cBo`2Le_l0100000003`3Ei^H0eN[V0=GViP3Ej^H0eNKV0=GZiP3Ei^H0eN[V0=GViP3Ej^H0
eNKV0=GZiP3Ei^H0eN[V000000080<F^a@D0000002d0J];f06[Fo`1Zd_H0J]Ko06[BmP1Ze_l0J];f
06[Fo`1Zd_H0J]Ko06[BmP1Ze_l0J];f06[Fo`1Zd_H0J]Ko06[BmP000000Y<kf0:CBmP2Tc_H0Y=;f
0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf
0:CBmP2Tc_H000000;g:gP35b^H0_L[N0<G:iP2mb]h00P0000005@35d^H0aLkN0<GBiP35c]h0aM;V
0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP000000aM;V0<G>gP35d^H0aLkN
0000000M0?ooo`030000003oool0oooo01L0oooo000:0?ooo`8000002P3oool2000000@0oooo00<0
00000?ooo`3oool08`3oool00`000000oooo0?ooo`0D0?ooo`<0000000d0Y<k^0:cBmP2Tc^h0[=;f
0:C>kP2/d_H0Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H0Y<k^00<0000001@0Y<k^0:cBmP2Tc^h0[=;f
0:C>kP2/d_H0Y<k^0:cBmP2Tc^h0[=;f0000002Ld_l0W=;f09cBo`2Ld_H0W=;o09cBmP2Ld_l0W=;f
09cBo`L0000001d0eN[V0<gViP3Ej^H0cNKV0=GZiP3=i^H0eN[V0<gViP3Ej^H0cNKV0=GZiP3=i^H0
eN[V0<gViP3Ej^H000000<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@000000
Lh7E07>5e@040000001206[BmP1Ze_H0J];f06[FmP1Zd_H0J]Kf06[BmP1Ze_H0J];f06[FmP1Zd_H0
J]Kf06[BmP1Ze_H0J];f0000002Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0
Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP000000_LKN0;g:iP2ma]h0
00000<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0
aLkN0<G>iP35c]h0aLkV00000035c^H0aLkN0<G>iP35c]h000006`3oool2000001T0oooo000K0?oo
o`8000009P3oool00`000000oooo0?ooo`0A0?ooo`8000000P2/a^h2000000090:C>mP2/d_H0Y<kf
0:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf00<0000000h0Y<kf0:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf
0:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf0:cBmP80000000<0W=;o09cFo`2Ld_l01P000000502Dk/d0
Rn[50000003Ei^H0eN[V0=GViP3Ej^H0eNKV0=GZiP3Ei^H0eN[V0=GViP3Ej^H0eNKV0=GZiP3Ei^H0
eN[V0=GViP3Ej^H000003035[/D00`000000LhGE07>5e@0307>5e@@0000002<0J]Ko06[BmP1Ze_l0
J];f06[Fo`1Zd_H0J]Ko06[BmP1Ze_l0J];f06[Fo`1Zd_H000000:C>mP2Td_H0Y<kf0:CBmP2Tc_H0
Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0
00000<G:iP020000000J0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0
aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP000000aLkN0<GBiP35c]h0aM;V0<G>gP00000L
0?ooo`030000003oool0oooo01H0oooo000K0?ooo`030000003oool0oooo02H0oooo00<000000?oo
o`3oool03@3oool30000000<0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0000002Tc^h0[=;f0:C>kP2/
d_H0Y<k^0`0000004P2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cB
mP2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cBmP@000000P1ch/d20000000308_Za@2;k/D0000000800000
02L0eN[V0<gViP3Ej^H0cNKV0=GZiP3=i^H0eN[V0<gViP3Ej^H0cNKV0=GZiP3=i^H0eN[V0<gViP3E
j^H000000<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@00
0000LhGE07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1e@1cQMD01P000000601Zd_H0J]Kf06[BmP1Z
e_H0J];f06[FmP1Zd_H000000:C>mP2Td_H0Y<kf0:CBmP2Tc_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmP2T
c_H0Y=;f0:C>mP2Td_H0Y<kf0:CBmPD0000001`0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>
gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0000
0035c]h0aLkV0<G>gP35c^H000007@3oool00`000000oooo0?ooo`0E0?ooo`006`3oool00`000000
oooo0?ooo`0W0?ooo`030000003oool0oooo00X0oooo0P00000:0:c6kP030000002Tc_H000000080
000001<0Y<kf0:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf0:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf0:cB
mP2Tc_H0[=;f0:C>mP2/d_H0Y<kf0080000000@0Ln;E07?Rc@1ch]D0Ln;=0P0000000`2Dk/d0Rn[5
000000020000000A0>KFa@3Ve/d000000000003Ei^H0eN[V0=GViP3Ej^H0eNKV0=GZiP3Ei^H0eN[V
0=GViP3Ej^H0eNKV0=GZiP0000003`35[/D00`000000LhGE07>5e@0:07>5e@0300000008A<D00000
00D0000000P0J]Ko06[BmP000000Y<kf0:CBmP2Tc_H0Y=;f0:C>mPX000000280Pl[o07_6o`23b_l0
NlKo08?:o`000000aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0
aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN00000035c]h0aM;V0<G>gP35d^H0
0000703oool00`000000oooo0?ooo`0E0?ooo`006`3oool2000002T0oooo00<000000?ooo`3oool0
1`3oool20000000<0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^
0`0000004`2/d_H0Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H0
Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H00P00000507?Rc@030000002;j/D00000008000000P3Nl]h0
1@000000i]K50=kBa@3Ve/D0g];50080000000P0eN[V0<gViP3Ej^H0cNKV0=GZiP3=i^H0eN[V0<gV
iP8000000280aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk5
0;f^a@35[/D0_Jk50000001cPMD0LhGE07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1e@1cQMD0Lh7E
07>5e@1cPMD000000014a@08A<D004C50P0000000`00NK@00000000000060000000Y07_6o`23a_l0
NlKo08?6o`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`000000
aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0
aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN00000035c]h00P0000000`35c^H000000?ooo`0J0?oo
o`8000005P3oool001/0oooo00<000000?ooo`3oool0:@3oool00`000000oooo0?ooo`030?ooo`<0
00002`2/a^h3000000<0[<K^014000000:cBmP2Tc_H0[=;f0:C>mP2/d_H0Y<kf0:cBmP2Tc_H0[=;f
0:C>mP2/d_H0Y<kf0:cBmP2Tc_H0[=;f0:C>mP020000000507?Re@1ch/d0Ln;E07?Rc@1ch]D01000
00040=kbgP080000003Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d2000000060=GViP3Ej^H0eNKV
0=GZiP3Ei^H000004`35[/D00`000000LhGE07>5e@0;07>5e@0400000008A<D024C500Q4a@<00000
02h007Vd001m_@00NK@0000007_6o`23b_l0NlKo08?:o`1ka_l0Pl[o07_6o`23b_l0NlKo08?:o`1k
a_l0Pl[o07_6o`23b_l0NlKo08?:o`1ka_l0Pl[o07_6o`23b_l000000<GBiP35c]h0aM;V0<G>gP35
d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35
c]h0aM;V0P0000001@35c]h0cM;V0<G>gP3=d^H0000001`0oooo00<000000?ooo`3oool04`3oool0
01/0oooo0P00000[0?ooo`80000000h0oooo000000000000[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H0
[<K^0:c2iP2/a^h0[<;V0:c6kP<0000000L0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^00800000
00`0Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H0Y<k^0:cBmP2Tc^h0[=;f0:C>kP2/d_H2000000D0Ln;=
0`0000060=kbgP0;0000003Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D00P00
0000:`3Ej^H0cNKV0000002m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk5
0<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50000001cQMD0Lh7E07>5e@1cPMD0LhGE
07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1e@1cQMD0000000Q4a@00A<D024C50014a@08A<D00`00
0002001i]080000002L0PlKo07_6o`23a_l0NlKo08?6o`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`1k
a_l0PlKo07_6o`23a_l0NlKo08?6o`1ka_l000000<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35
c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H00`0000001@35
c^H0aLkN0<G>iP35c]h0000001`0oooo00<000000?ooo`3oool04`3oool001/0oooo00<000000?oo
o`3oool0:P3oool3000000X0[<K^0`00000<0:c6kP0:0000002Tc_H0[=;f0:C>mP2/d_H0Y<kf0:cB
mP2Tc_H0[=;f0:C>mP80000000@0Ln;E07?Rc@1ch]D0Ln;=100000070=kbgP0>0000003Ve/d0i]K5
0>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d2000001H0aJk500<00000
07>5e@1cQMD02`1cQMD00`00000024C500Q4a@0300Q4a@0f00000000B;@0000000000000NK@007fm
00000023b_l0NlKo08?:o`1ka_l0Pl[o07_6o`23b_l0NlKo08?:o`1ka_l0Pl[o07_6o`23b_l0NlKo
08?:o`1ka_l0Pl[o07_6o`000000aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V
0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H000000<gBiP35c]h0cM;V0000003=d^H0aLkN
0<gBiP35c]h00000703oool00`000000oooo0?ooo`0B0?ooo`00203oool2000000@0oooo0P000003
0?ooo`D000000P3oool00`000000oooo0?ooo`0X0?ooo`<000000P2T^^H020000000[<K^0:c2iP2/
a^h0[<;V0:c6kP2/`^H0[<K^0`0000005`2/a^h0[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6
kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H000000:C>kP2/d_H0Y<k^0:cBmP2Tc^h0[=;f0080
00000P1ch/d4000000030;C^kP000000g_;N00P0g_;N03l000000=kBa@3Ve/D0g];50>KFa@3Nd/D0
i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@000000eKJd0000002m[/D0aJk50;f^a@35[/D0
_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0
aJk50;f^a@000000Lh7E07>5e@1cPMD0LhGE07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1e@1cQMD0
Lh7E00000000A<D024C50014a@08A<D004C500Q4a@00000004Rd0014]00000000P00NK@08`000000
PlKo07_6o`23a_l0NlKo08?6o`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`1ka_l0PlKo07_6o`23a_l0
NlKo08?6o`1ka_l000000<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0
aLkV0<G>gP35c^H0aLkN0080000000/0aLkV0<G>gP35c^H0aLkN00000035c]h0aLkV0<G>gP35c^H0
aLkN0000000L0?ooo`030000003oool0oooo00D0oooo1@0000020?ooo`8000000`3oool000L0oooo
00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`020?ooo`8000009P3oool300000007
0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^000000030:c6kP<00000502/a^h2000000030:cBmP2Tc_H0
000000D0000000D0_O;f0;g^mP2ml_H0_Nkf000000090=kbgP0B0000003Ve/D0i]K=0>KFa@3Ve/d0
i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0000003N^[d000005P35[/D00`00
0000LhGE07>5e@0;07>5e@0300000008A<D024C500@024C50P000000=000A;@0000000000000OKd0
000008?:o`1ka_l0Pl[o07_6o`23b_l0NlKo08?:o`1ka_l0Pl[o07_6o`23b_l0NlKo08?:o`1ka_l0
Pl[o07_6o`23b_l0NlKo00000035c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0
aLkN0<GBiP35c]h000000<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP000000aLkN0<gBiP35c]h0
cM;V0<G>gP00000J0?ooo`8000002@3oool01@000000oooo0?ooo`3oool000000080oooo0@000001
0?ooo`40oooo00070?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`0S0?ooo`8000002P2T^^H30000000I0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V
0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^
0:c2iP2/a^h0[<;V0:c6kP03000000090;gbmP2dk^h0_O;f0;C^kP2ml_H0]>k^0;gbmP2dk^h00000
00T0g_;N04<000000=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K5
0=kBa@3Ve/D000000=Fj_@3E][@0eKZm00000035[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk5
0<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@000000LhGE
07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1e@00000004C5
00Q4a@00A<D024C50014a@08A<D004C500000000A;@004Rd00<0000001d0PlKo07_6o`23a_l0NlKo
08?6o`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`1ka_l00000
0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP020000000?0<G>gP35c^H0aLkN
0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN00000035c]h0aLkV0<G>gP35c^H0000001`0oooo00<00000
0?ooo`3oool01P3oool01@000000oooo0?ooo`3oool000000080oooo0@0000010?ooo`40oooo0000
0`3oool000000000000200000080oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?oo
o`020?ooo`8000008@3oool30000000:0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^
0:BjiP80000000<0[;[^0000002/a^h05`2/a^h20000000;0;gbmP2mk_H0_O;f0;g^mP2ml_H0_Nkf
0;gbmP2mk_H0_O;f0;g^mP0000002@3Nl]h05@000000i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K5
0>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@000000g[Zm0=Fj_@3N^[d0eKZm0000000F0<F^a@03
0000001cQMD0LhGE00/0LhGE00<0000000Q4a@08A<D01P08A<D01000000004Bd0018]000A;@30000
000]07_6o`23b_l0NlKo08?:o`1ka_l0Pl[o07_6o`23b_l0NlKo08?:o`1ka_l0Pl[o07_6o`23b_l0
NlKo08?:o`1ka_l000000<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H000000<gBiP35c]h0
cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN00000035c]h0cM;V0<G>gP3=d^H0
000001`0oooo00<000000?ooo`3oool01@3oool01@000000oooo0?ooo`3oool000000080oooo0@00
00010?ooo`40oooo00070?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3o
ool00`000000oooo0?ooo`0M0?ooo`<000002P2T^^H3000000@0Y;[V01H000000:c6kP2/`^H0[<K^
0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V
0:c6kP2/`^H0[<K^0P000000302dk^h0_O;f0;C^kP2ml_H0]>k^0;gbmP2dk^h0_O;f0;C^kP2ml_H0
]>k^000000X0g_;N04<000000>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3V
e/D0g];50>KFa@3Nd/D000000=Ff]03E^[d0eKJd0=Fj_@3E][@000000;f^a@35[/D0_Jk50<F^a@2m
[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35
[/D0_Jk50000001cPMD0LhGE07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1e@1cQMD0Lh7E07>5e@1c
PMD0LhGE00000008A<D004C500Q4a@00A<D024C50014a@08A<D004C50080000000<004Rd0014]000
B;@00P0000005`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`1ka_l0PlKo07_6o`23a_l0NlKo08?6o`1k
a_l0PlKo07_6o`23a_l000000<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0080000001@0aLkV0<G>gP35
c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV00000035c^H0aLkN0<G>iP35
c]h0aLkV000001/0oooo00<000000?ooo`3oool01@3oool01@000000oooo0?ooo`3oool000000080
oooo0@0000010?ooo`40oooo00080?ooo`8000002P3oool2000000@0oooo00<000000?ooo`3oool0
6P3oool30000000:0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP<0000000T0
Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP0000004`2/a^h03P000000_O;f0;g^mP2ml_H0
_Nkf0;gbmP2mk_H0_O;f0;g^mP2ml_H0_Nkf0;gbmP2mk_H000002P3Nl]h05`000000i]K=0>KFa@3V
e/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@000000eKZm0=jj_@3E
^[d0g[Zm0=Fj_@3N^[d0000001H0aJk500<0000007>5e@1cQMD02`1cQMD00`00000024C500Q4a@06
00Q4a@0700000008D<d000000018]000A;@004Rd0014]0050000000Y07_6o`23b_l0NlKo08?:o`1k
a_l0Pl[o07_6o`23b_l0NlKo08?:o`1ka_l0Pl[o00000035d^H0aLkN0<GBiP35c]h000000<G>gP3=
d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP00
0000cM;V0<G>gP3=d^H0aLkN0<gBiP0000006P3oool2000000D0oooo0P0000040?ooo`8000000`3o
ool001T0oooo0P00000I0?ooo`<000000P2L]^H00`000000Y;[V0:BjiP050:BjiP<00000302T^^H2
0000000?0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H0[<K^0:c2iP2/
a^h0[<;V0080000000d0_O;f0;C^kP2ml_H0]>k^0;gbmP2dk^h0_O;f0;C^kP2ml_H0]>k^0;gbmP2d
k^h0000000X0g_;N02h000000>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3V
e/D0g];50>KFa@3Nd/D000000=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d000000<F^a@2m
[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35
[/D0_Jk50<F^a@2m[/D0aJk5100000007`1cPMD0LhGE07>1e@1cQMD0Lh7E07>5e@1cPMD0LhGE07>1
e@1cQMD0Lh7E00000000A<D024C50014a@08A<D004C500Q4a@00A<D024C500000008D<d000000018
]000A;@004Rd0014]0000000FZGf05ZZmP1JYOH02P0000001@23a_l0NlKo08?6o`000000aLkV0080
000001P0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>
iP35c]h0aLkV0<G>gP35c^H000000<G>iP35c]h0aLkV0<G>gP00000G0?ooo`D00000403oool001T0
oooo00<000000?ooo`3oool05@3oool30000000:0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0000002T
^^H0[;[^0:BjiP<000000180Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:Bj
iP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP00000<0:c6kP80000000`0_Nkf0;gbmP2mk_H0_O;f0;g^
mP2ml_H0_Nkf0;gbmP2mk_H0_O;f0;g^mP2ml_H2000000X0g_;N01/000000>KFc@3Ve/D0i]K=0>KF
a@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0000003N^[d0eKZm0=jj
_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@0000004`35[/D20000000308=i_@2;OLD0PgVm00P00000
0`1cQMD00`00000024C500Q4a@0600Q4a@0300000008D<d0253=00800000010004Bd0018]000A;@0
000005ZUmP1JZ_l0FZGf05ZZo`1JYOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`1JYOH0FZ[o1@0000006`3=
d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35
c]h0cM;V0<G>gP3=d^H0aLkN00000035c]h0cM;V0<G>gP3=d^H0aLkN0000000B0?ooo`@00000103o
ool00`000000oooo0?ooo`0>0?ooo`006@3oool200000180oooo1000000:09bfiP<000005@2T^^H0
2P000000[<;V0:c6kP2/`^H0[<K^0:c2iP2/a^h0[<;V0:c6kP2/`^H20000000>0;gbmP2dk^h0_O;f
0;C^kP2ml_H0]>k^0;gbmP2dk^h0_O;f0;C^kP2ml_H0]>k^0;gbmP00000<0=kbgP0/0000003Nd/D0
i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50000003E^[d0
eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@000000;f^a@35[/D0_Jk50<F^a@2m[/D0
aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D2000000030;BI_@2d
UK@0000000X0PgVm1@0000002@08A<D004C500Q4a@00A<D024C50014a@08A<D004C50000000300Q@
c@0a00000000A;@004Rd0014]0000000FZGf05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0FZGf05ZZmP1JYOH0
FZ[f05ZUmP1JZ_H0FZGf05ZZmP1JYOH0FZ[f00000035c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0
aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN00000035c]h0
aLkV0<G>gP35c^H0aLkN0000000>0?ooo`<00000203oool00`000000oooo0?ooo`0>0?ooo`006@3o
ool00`000000oooo0?ooo`0>0?ooo`<0000000/0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:Bf
kP2L]^H0Y;K^09bfiP030000000J09bfiP000000Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bj
kP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:Bj
iP0000070:c6kP0@0000002mk_H0_O;f0;g^mP2ml_H0_Nkf0;gbmP2mk_H0_O;f0;g^mP2ml_H0_Nkf
0;gbmP2mk_H0_O;f000000`0g_;N01d000000>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0
i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d000000=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0
g[Zm0=Fj_@3N^[d0eKZm0000000?0<F^a@030000002dVKd0]9Vm0080]9Vm00l0000008]ma@23NKd0
Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd00P00000800Q4a@0a
00000008D<d0253=00Q@c@00000004Bd0018]000A;@0000005ZUmP1JZ_l0FZGf05ZZo`1JYOH0FZ[o
05ZUmP1JZ_l0FZGf05ZZo`1JYOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`000000cM;V0<G>gP3=d^H0aLkN
0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V
0<G>gP000000aLkN0<gBiP02000000030<G>gP000000oooo00T0oooo0`00000<0?ooo`030000003o
ool0oooo00d0oooo000I0?ooo`030000003oool0oooo00X0oooo1000000<09bfiP8000001@2L]^H0
0`000000Y;[V0:BjiP0E0:BjiP050000002/`^H0[<K^0:c2iP2/a^h00P0000003`2dk^h0_O;f0;C^
kP2ml_H0]>k^0;gbmP2dk^h0_O;f0;C^kP2ml_H0]>k^0;gbmP2dk^h0_O;f0000000<0=kbgP0c0000
003Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kB
a@000000eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d000000<F^
a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@2m[/D000000;BE
]02dVKd0]9Fd0;BI_@2dUK@0000000l0PgVm0`0000001`00A<D024C50014a@08A<D004C500Q4a@00
00001008D<d20000000X0018]000A;@0000005ZUmP1JZ_H0FZGf05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0
FZGf05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0FZGf00000035c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0
aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP800000
00@0aLkN0<G:gP35c]h000001`3oool3000000l0oooo00<000000?ooo`3oool03@3oool001T0oooo
0P0000060?ooo`D0000000d0Rk[f08^fmP2;^_H000000:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:Bf
kP2L]^H0Y;K^00<000000280Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP000000Y;[V0:bj
kP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:Bj
iP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP000000[<K^0P000000402ml_H0_Nkf0;gbmP2mk_H0_O;f
0;g^mP2ml_H0_Nkf0;gbmP2mk_H0_O;f0;g^mP2ml_H0_Nkf0;gbmP00000<0=kbgP0P0000003Ve/D0
i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@000000
eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@00000<
0<F^a@030000002dVKd0]9Vm00D0]9Vm01<0000008=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;
OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD00000059Qa@1BGLD0000000H024C500<0000000Q@c@08
D<d00`08D<d09`00000004Rd0000001JZ_l0FZGf05ZZo`1JYOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`1J
YOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`1JYOH000000<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=
d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP02000000060<g>iP35
c]h0cLkV0<G>gP3=c^H000000P3oool400000180oooo00<000000?ooo`3oool03@3oool001P0oooo
00<000000?ooo`3oool01P0000002P2;^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0RkK^08^jmP00
000509bfiP<00000302L]^H00`000000Y;[V0:BjiP0D0:BjiP@0000000l0]>k^0;gbmP2dk^h0_O;f
0;C^kP2ml_H0]>k^0;gbmP2dk^h0_O;f0;C^kP2ml_H0]>k^0;gbmP0000003@3Nl]h0:P000000i]K5
0=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@000000eKJd
0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm
00000035[/D0_Jk50<F^a@2m[/D0aJk50;f^a@35[/D0_Jk50<F^a@80000000T0]9Fd0;BI_@2dUK@0
]9Vm0;BE]02dVKd0]9Fd0;BI_@0000003P23NKd02`000000DUfm059Qa@1BGKd000000014a@08A<D0
04C500Q4a@00A<D0000000H0253=0`000000:`1JZ_H0FZGf05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0FZGf
05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0FZGf05ZZmP1JYOH000000<G>gP35c^H0aLkN0<G>iP35c]h0aLkV
0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h000000<G:gP35c]h00000
0<G>gP35b]h0aLkN0<G:gP35c]h00P00000F0?ooo`030000003oool0oooo00d0oooo000A0?ooo`80
00001@3oool30000000B08^fmP2;^_H0RkKf08^jmP2;]_H0Rk[f08^fmP2;^_H0RkKf08^jmP2;]_H0
Rk[f08^fmP2;^_H0RkKf08^jmP000000Y;K^0`000000902T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T
]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP000000[;[^0:BjiP2/^^h0Y;[V0:bjkP2T
^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP80
000000@0_MKf0;CBkP2me_H0]=;^100000002P2ml_H0_Nkf0;gbmP2mk_H0_O;f0;g^mP2ml_H0_Nkf
0;gbmP00000=0=kbgP0R0000003Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0
i]K=0>KFa@3Ve/d0i]K50>KFc@000000g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0
eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm000000P0aJk500<000000;BI_@2dVKd0202dVKd05000
0000PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23
NKd00000059Ma@1BHLD0DUg50P00000400Q4a@0300000008D<d0253=00D0253=0P0000007`1JZ_l0
FZGf05ZZo`1JYOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`1JYOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`1JYOH0
00000<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP3=d^H0
0P0000002P35c]h0cLkV0<G>gP3=c^H000000<g>iP35c]h0cLkV0<G>gP00000G0?ooo`030000003o
ool0oooo00d0oooo000@0?ooo`040000003oool0oooo000000@0oooo0P000000402;]^h0Rk[f08^f
kP2;^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H3000001<0
W;KV00<000000:BjiP2T^^H03`2T^^H2000000X0]=;^1@000000102ml_H0]>k^0;gbmP00000=0=kb
gP0h0000003Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0
g];50>KFa@000000eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0
eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@000000;f^a@35[/D0_Jk50<F^a@2m[/D0aJk50000002dVKd0
]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]000000>08=i_@0;0000001B
HLD0DUfm059Qa@1BGKd0DV7500000008A<D004C500Q4a@0000002008D<d20000000[05ZZmP1JYOH0
FZ[f05ZUmP1JZ_H0FZGf05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0FZGf05ZZmP1JYOH0FZ[f00000035c^H0
aLkN0<G>iP35c]h0aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP35c^H000000<G>gP35b]h0
aLkN0<G:gP35c]h0aL[N0<G>gP000000aLkN0<G:gP35c]h0aL[N0000000F0?ooo`030000003oool0
oooo00d0oooo000@0?ooo`040000003oool0oooo000000@0oooo00l000000?ooo`000000RkKf08^j
mP2;]_H0Rk[f08^fmP2;^_H0RkKf08^jmP2;]_H0Rk[f08^fmP2;^_H00`0000009P2;^_H0RkKf08^j
mP000000Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bf
iP2T]^h0W;KV0:BfkP2L]^H0Y;K^0000002/^^h0Y;[V0:bjkP2T^^H0[;[^0:BjiP2/^^h0Y;[V0:bj
kP2T^^H0[;[^0:BjiP2/^^h0Y;[V0P0000004@2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^
0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf00H000002P3Nl]h09@000000i]K=0>KF
a@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d000000=jj
_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj
_@3N^[d0eKZm0=jj_@0000001@35[/D00`000000]9Vm0;BI_@0;0;BI_@0G0000002;OLD0PgVm08]m
a@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm0000001BGLD0DV75059M
a@1BHLD0DUg5059Qa@0000000`08A<D00`000000253=00Q@c@0600Q@c@80000001X0FZ[o05ZUmP1J
Z_l0FZGf05ZZo`1JYOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`1JYOH0FZ[o05ZUmP1JZ_l000000<gBiP35
c]h0cM;V0<G>gP3=d^H0aLkN0<gBiP35c]h0cM;V0<G>gP80000000l0cLkV0<G>gP3=c^H0aLkN0<g>
iP35c]h0cLkV0<G>gP000000aLkN0<g>iP35c]h0cLkV0<G>gP0000005@3oool00`000000oooo0?oo
o`0=0?ooo`00403oool010000000oooo0?ooo`0000040?ooo`80000000X0oooo000000000000Rk[f
08^fkP2;^_H0RkK^08^jmP2;]^h0Rk[f0`000000202;^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0
00004`2L]^H00`000000Y;[V0:BjiP090:BjiP800000502dd^h01@000000eM[N0<gFgP3Ef]h0cMKN
00H00000103Nl]h0:0000000g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kB
a@3Ve/D0g];50>KFa@3Nd/D000000=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj
_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d000000<F^a@2m[/D20000000?
0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@00000
00l0PgVm00L00000059M_@1BHLD0DUfm059Qa@1BGKd0DV750080000000<004C500Q4a@0000002@08
D<d30000000Y05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0FZGf05ZZmP1JYOH0FZ[f05ZUmP1JZ_H0FZGf05ZZ
mP000000aLkV0<G>gP35c^H0aLkN0<G>iP35c]h0aLkV0<G>gP000000aL[N0<G>gP35b]h0aLkN0<G:
gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP000000aL[N0<G>gP35b]h0aLkN0<G:gP0000005@3oool0
0`000000oooo0?ooo`0<0?ooo`00403oool010000000oooo0?ooo`0000040?ooo`030000003oool0
oooo0080oooo00D0000008^jmP2;]_H0Rk[f08^fmP020000000X08^jmP2;]_H0Rk[f08^fmP2;^_H0
RkKf08^jmP2;]_H0Rk[f08^fmP2;^_H000000:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0
Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP000000[;[^0:BjiP2/^^h0
Y;[V0:bjkP2T^^H0[;[^0:BjiP8000000240_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2m
e_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H000000=GJgP3=
f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h010000000:03Ve/D0i]K=0>KFa@3V
e/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@000000eKZm0=jj_@3E
^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N
^[d0eKZm0=jj_@3E^[d000000<F^a@00000A0;BI_@0K00000023NKd0Rgg508=i_@2;OLD0PgVm08]m
a@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg50000001BHLD0DUg5059Qa@1BGLD0DV75059M
a@1BHLD0DUg500000008A<D0000000X0253=00<0000006:EkP000000100000003`1JZ_l0FZGf05ZZ
o`1JYOH0FZ[o05ZUmP1JZ_l0FZGf05ZZo`000000cM;V0<G>gP3=d^H0aLkN0<gBiP020000000C0<G>
gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP000000aLkN0<g>
iP35c]h0cLkV0000000E0?ooo`030000003oool0oooo00`0oooo000A0?ooo`800000103oool00`00
0000oooo0?ooo`040?ooo`@0000000h0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0RkK^08^jmP2;]^h0
Rk[f08^fkP2;^_H0RkK^000001@0W;KV00<000000:BjiP2T^^H00`2T^^H2000001H0]=;^01400000
0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h00000
0=k:a@020000000h0>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];50>KFa@3Nd/D0i]K50=kBa@3Ve/D0g];5
0000003E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm
0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0000002dUK@0]9Vm0;BE]02dVKd0]9Fd
0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@00000?08=i
_@0<0000001BHLD0DUfm059Qa@1BGKd0DV75059M_@1BHLD0DUfm00000000A<D000002@08D<d01`00
0000FY7V06:EkP1JTNH0HYG^05ZAiP1RUNh0100000008@1JYOH0FZ[f05ZUmP1JZ_H0FZGf00000035
c]h0aLkV0<G>gP35c^H000000<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35
c]h0aL[N0<G>gP35b]h0aLkN00000035c]h0aL[N0<G>gP35b]h0aLkN0000000D0?ooo`030000003o
ool0oooo00`0oooo000G0?ooo`8000000`3oool40000000X08^jmP2;]_H0Rk[f08^fmP2;^_H0RkKf
08^jmP2;]_H0Rk[f08^fmP2;^_H0RkKf08^jmP2;]_H0Rk[f08^fmP000000W;KV0:BfkP2L]^H0Y;K^
09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV
0:BfkP000000[;[^0:BjiP80000001H0_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0
]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0P0000004`3E
f]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0000003N
c/D0g/k=0=k>a@3Nc/d00P0000008@3Ve/D0i]K=0>KFa@3Ve/d0i]K50>KFc@3Ve/D0i]K=0>KFa@3V
e/d000000=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E
^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@0200000180]9Vm01X0000008=i_@2;OLD0
PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm0000001BGLD0
DV75059Ma@1BHLD0DUg5059Qa@1BGLD0DV75059Ma@8000002P08D<d02P000000FY7^06:EkP1JTNh0
HYG^05ZAkP1RUNh0FY7^06:EkP1JTNh50000000305ZUmP000000aLkN0080000001P0cLkV0<G>gP3=
c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP00
0000cLkV0<G>gP3=c^H0aLkN0<g>iP00000C0?ooo`030000003oool0oooo00`0oooo000G0?ooo`03
0000003oool0000000800000103oool04P000000Rk[f08^fkP2;^_H0RkK^08^jmP2;]^h0Rk[f08^f
kP2;^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0RkK^000001<0W;KV0`00000G0;CBkP0H0000003=
e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3E
f]h000000=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50P000000=@3Ve/D0g];50>KFa@3Nd/D0i]K50=kB
a@3Ve/D000000=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj
_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]0000000cJZd0<f^_@000000]9Vm0;BE
]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI
_@2dUK@0000000h0PgVm00d00000059Qa@1BGKd0DV75059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M
_@1BHLD0000000X0253=00h0000005ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^05ZAiP1RUNh0FY7V06:E
kP1JTNH0HYG^05ZAiP<0000001X0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN
0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP000000aLkN0<G:gP35c]h0aL[N
000001<0oooo00<000000?ooo`3oool0303oool001L0oooo0P0000080?ooo`0Q0000002;^_H0RkKf
08^jmP2;]_H0Rk[f08^fmP2;^_H0RkKf08^jmP2;]_H0Rk[f08^fmP2;^_H0RkKf08^jmP2;]_H00000
09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^09bfiP2T]^h0W;KV
00<0000003@0Y;k^0:C2kP00000000000;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^
0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0000003=f]h0eM[N
0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h00000
0=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0P000000803Ve/D0i]K=0>KFa@3Ve/d0
00000=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0
g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d000000<f^_@3=Z[@000004P2dVKd06`00
0000Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;
OLD00000059Qa@1BGLD0DV75059Ma@1BHLD0DUg5059Qa@1BGLD0DV75059Ma@03000000T0253=02/0
000005ZAkP1RUNh0FY7^06:EkP1JTNh0HYG^05ZAkP1RUNh0FY7^06:EkP1JTNh0HYG^05ZAkP1RUNh0
FY7^00000035c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0
cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0000003=c^H0aLkN0<g>iP35c]h000000180oooo00<0
00000?ooo`3oool0303oool001P0oooo00<000000?ooo`3oool01`3oool04P000000Rk[f08^fkP2;
^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0RkK^000000/0
W;KV0`000000202L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h000004`2dd^h07P000000cMKN0=GJ
gP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0000
003Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@8000000340i]K5
0=kBa@000000eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd
0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0000003=[[d0cJZd0<f^_@000000]9Vm
0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd
0;BI_@2dUK@0000000l0PgVm00h00000059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M_@1BHLD0DUfm
059Qa@1BGKd0000004YYc@8000001`08D<d0:0000000HYG^05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^
05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^05ZAiP1RUNh000000<G>gP35b]h0aLkN0<G:gP35c]h0aL[N
0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h00000
0<G:gP35c]h2000000030<G:gP000000oooo0100oooo00<000000?ooo`3oool0303oool001T0oooo
0P0000080?ooo`80000001H0RkKf08^jmP2;]_H0Rk[f08^fmP2;^_H0RkKf08^jmP2;]_H0Rk[f08^f
mP2;^_H0RkKf08^jmP2;]_H0000009bfiP2T]^h0W;KV0:BfkP2L]^H0Y;K^10000000?P2T_^h0Y<;^
0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0000002dd^h0_MKf0;CBkP2me_H0]=;^
0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP2me_H0]=;^0000003=f]h0eM[N
0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h00000
0=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=
0P0000007@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0
g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d000000<f^_@3=Z[@0cJjm0<fZ]03=[[d0
00000180]9Vm01l0000008=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0
Rgg508=i_@2;OLD00000059Qa@1BGLD0DV75059Ma@1BHLD0DUg5059Qa@1BGLD0DV75059Ma@1BHLD0
000004YYe@1:JLd0BVWE008000001P08D<d09@000000HYG^05ZAkP1RUNh0FY7^06:EkP1JTNh0HYG^
05ZAkP1RUNh0FY7^06:EkP1JTNh0HYG^05ZAkP1RUNh000000<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN
0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h00P00
0000103=c^H0aLkN0<g>iP00000A0?ooo`030000003oool0oooo00`0oooo000K0?ooo`030000003o
ool0oooo00L0oooo0140000008^fkP2;^_H0RkK^08^jmP2;]^h0Rk[f08^fkP2;^_H0RkK^08^jmP2;
]^h0Rk[f08^fkP2;^_H0RkK^0000000209bfiP<000000100Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0
Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0P00000>0;CBkP1D0000003=e]h0
eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0
00000=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0
g/k=0=k:a@000000eKjm0000003E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0
eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0000003=[[d0cJZd0<f^_@3=Z[@0
cJjm0<fZ]0000000]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0
]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd000003`23NKd03@000000DV75059M_@1BHLD0DUfm059Qa@1B
GKd0DV75059M_@1BHLD0DUfm059Qa@000000101:JLd2000000D0253=02/0000006:EkP1JTNH0HYG^
05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^00000035c]h0aL[N
0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN
0<G:gP000000aL[N00000035b]h0aLkN0<G:gP35c]h000000140oooo00<000000?ooo`3oool02`3o
ool001`0oooo00<000000?ooo`3oool01`3oool02@000000RkKf08^jmP2;]_H0Rk[f08^fmP2;^_H0
RkKf08^jmP090000001U0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0
Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0000002me_H0]=;^0;gFmP2dd^h0
_MKf0;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0;CBkP000000cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0
cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP000000g/k50=k>c@3Nc/D0
g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0000003N`/D0
00000=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0
g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d000000<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]0000000
4@2dVKd090000000PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0
PgVm08]ma@23NKd00000059Ma@1BHLD0DUg5059Qa@1BGLD0DV75059Ma@1BHLD0DUg5059Qa@1BGLD0
000004YYc@1:JMD0BVW=04YYe@1:JLd0BVWE000000@0253=0240000005ZAkP1RUNh0FY7^06:EkP1J
TNh0HYG^05ZAkP1RUNh0FY7^06:EkP1JTNh0HYG^05ZAkP1RUNh0FY7^06:EkP000000cLkV0<G>gP3=
c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H00P0000002@35
c]h0cLkV0000003=c^H0aLkN0<g>iP35c]h0cLkV0000000@0?ooo`030000003oool0oooo00/0oooo
000M0?ooo`030000003oool0oooo00L0oooo200000007`1c/_l0JZkf07>bo`1Z[_H0Lk;o06Z^mP1c
/_l0JZkf0000002L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L
_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h00P0000080;CBkP80000005H0cMKN
0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N
0<gFgP000000g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[5
0=k>c@3Nb/D0g/k=0000003N`/D0eKjm0=k2a@000000eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm
0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@000000<fZ]03=[[d0cJZd
0<f^_@3=Z[@0cJjm0<fZ]03=[[d000000;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm
0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd000000h0PgVm00h00000059Qa@1BGKd0
DV75059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M_@1BHLD0DUfm000000H0BVW=0P00000300Q@c@0[
0000001JTNH0HYG^05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^
05ZAiP000000aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N
0<G>gP000000aLkN0<G:gP35c]h0aL[N0<G>gP000000aLkN0<G:gP35c]h0aL[N0000000@0?ooo`03
0000003oool0oooo00/0oooo000N0?ooo`800000203oool20000001c07>bo`1Z/_H0Lk;o06ZbmP1c
/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f07>bo`1Z/_H000000:BnkP2T`^h0Y;k^0:C2kP2T
_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T
`^h0Y;k^0:C2kP2T_^h000000;CBkP2me_H0]=;^0;gFmP2dd^h0_MKf0000003Ef]h0cM[N0=GJgP3=
f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP00
0000g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3N
c/D0g/k=0000003N`/D0eKjm0=k2a@3E_[d0g/;50000003N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N
^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d000000<fZ]03=[[d0cJZd0<f^_@3=
Z[@0cJjm0<fZ]03=[[d0cJZd0000000B0;BI_@0U0000002;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;
OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@000000DV75059Ma@1BHLD0DUg5059Qa@1B
GLD0DV75059Ma@1BHLD0DUg5059Qa@000000BVWE04YYc@1:JMD0BVW=04YYe@1:JLd0BVWE04YYc@02
00000080253=01`0000005ZAkP1RUNh0FY7^06:EkP1JTNh0HYG^05ZAkP1RUNh0FY7^06:EkP1JTNh0
HYG^05ZAkP1RUNh0FY7^00000035c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0
aLkN0P0000003@3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H000000<g>iP35c]h0cLkV0<G>gP00
00003`3oool00`000000oooo0?ooo`0;0?ooo`00803oool00`000000oooo0?ooo`070?ooo`0X0000
001c/_l0JZkf07>bo`1Z[_H0Lk;o06Z^mP1c/_l0JZkf07>bo`1Z[_H0Lk;o06Z^mP1c/_l0JZkf0000
002L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bn
kP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^000000@0]=;^05X000000=GJgP3=e]h0eM[N
0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN
0000003Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=
0=k:a@3Nc/d0g/[50000003E_[d0g/;50=Fn_@3N`/D0eKjm0000003E][@0eKZm0=Ff]03E^[d0eKJd
0=Fj_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]0000000cJZd0<f^_@3=Z[@0cJjm
0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]0000000]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd
0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]000000?08=i_@0>0000001BGKd0
DV75059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M_@1BHLD0DUfm059Qa@00000904YYc@80000001X0
253=0000001JTNH0HYG^05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^05ZAiP1RUNh0FY7V06:EkP1JTNH0
HYG^05ZAiP000000aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP8000000100aL[N0<G>gP35
b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN00000035c]h0aL[N0<G>gP35b]h000003P3oool0
0`000000oooo0?ooo`0;0?ooo`008@3oool00`000000oooo0?ooo`070?ooo`0W0000001c/_l0J[;f
07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f0000002T_^h0Y<;^
0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^
0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0080000004X0_MKf0000003Ef]h0cM[N0=GJgP3=f]h0eM[N
0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP000000g/k5
0=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=
0=k>a@000000eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003E^[d0g[Zm0=Fj_@3N^[d0eKZm
0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0000003=[[d0cJZd0<f^_@3=Z[@0cJjm
0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm00000180]9Vm02P0000008=i_@2;OLD0PgVm08]ma@23NKd0
Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD00000059Qa@1BGLD0DV75059Ma@1BHLD0
DUg5059Qa@1BGLD0DV75059Ma@1BHLD0DUg50000001:JLd0BVWE04YYc@1:JMD0BVW=04YYe@1:JLd0
BVWE04YYc@1:JMD0BVW=0P000000:P1RUNh0FY7^06:EkP1JTNh0HYG^05ZAkP1RUNh0FY7^06:EkP1J
TNh0HYG^05ZAkP1RUNh0FY7^06:EkP1JTNh000000<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0000003=
c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP000000aLkN0<g>iP35
c]h0cLkV000000h0oooo00<000000?ooo`3oool02`3oool00280oooo00<000000?ooo`3oool01`3o
ool20000002306Z^mP1c/_l0JZkf07>bo`1Z[_H0Lk;o06Z^mP1c/_l0JZkf07>bo`1Z[_H0Lk;o06Z^
mP000000W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:Bn
kP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h000000=GJgP3=e]h0eM[N0<gF
gP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJ
gP000000g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:
a@3Nc/d0g/[50000003E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@000000eKJd0=Fj
_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0=Ff]03E^[d000000<f^_@3=Z[@0cJjm0<fZ
]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0000002dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI
_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@0000003`23NKd0
3P000000DV75059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M_@1BHLD0DUfm059Qa@1BGKd00000301:
JLd50000000@06:EkP1JTNH0HYG^05ZAiP1RUNh0FY7V06:EkP1JTNH0HYG^05ZAiP1RUNh000000<G>
gP35b]h0aLkN0<G:gP80000001@0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N
0<G>gP35b]h0aLkN0<G:gP000000aL[N0<G>gP35b]h0aLkN000000d0oooo00<000000?ooo`3oool0
2`3oool002<0oooo0P0000090?ooo`0T0000001Z/_H0Lk;o06ZbmP1c/_l0J[;f07>bo`1Z/_H0Lk;o
06ZbmP1c/_l0J[;f07>bo`1Z/_H000000:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^
0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0P000000B`2mb^H0
00000000003=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0
eM[N0<gJgP3Ef]h000000=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0
g/k50=k>c@3Nc/D0g/k=0=k>a@000000eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0
g/;50000003N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0000003=[[d0
cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0000000A0;BI
_@0_00000023NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0
Rgg508=i_@000000DUg5059Qa@1BGLD0DV75059Ma@1BHLD0DUg5059Qa@1BGLD0DV75059Ma@1BHLD0
000004YYe@1:JLd0BVWE04YYc@1:JMD0BVW=04YYe@1:JLd0BVWE04YYc@1:JMD0BVW=04YYe@000000
NiWV07>IiP1kVNH02@0000001@1RUNh0FY7^06:EkP000000cLkV0080000001L0aLkN0<g>iP35c]h0
cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP000000
aLkN0<g>iP35c]h0cLkV0000000=0?ooo`030000003oool0oooo00X0oooo000U0?ooo`030000003o
ool0oooo00L0oooo00h0000006Z^mP1c/_l0JZkf07>bo`1Z[_H0Lk;o06Z^mP1c/_l0JZkf07>bo`1Z
[_H0Lk;o06Z^mP8000000140Y;k^09bnkP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:Bn
kP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP02000000050;C6iP2mb^H0]<KV0;g:iP2da^H00P000000
F@3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h0cMKN0=GJgP3=e]h0eM[N0<gFgP3Ef]h000000=k>
c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:
a@3Nc/d000000=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D000000=Fj
_@3E][@0eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@0eKZm0000003=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ
]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@000000]9Vm0;BE]02dVKd0]9Fd0;BI
_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0000000l0
PgVm00h00000059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M_@1BHLD0DUfm059Qa@1BGKd0DV750000
00`0BVW=00d0000007>EiP1cVNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1cUNH0LiWV
00D0000001T0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN
0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP000000aL[N0<G>gP35b]h0000000d0oooo00<00000
0?ooo`3oool02P3oool002H0oooo00<000000?ooo`3oool01`3oool20000000L07>bo`1Z/_H0Lk;o
06ZbmP1c/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f07>bo`000000Y<;^0:BnkP2T`^h0Y;k^
0:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h0Y<;^0:BnkP80000000T0_L[V0;C6iP2mb^H0
]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H00P000000A@3=f]h0eM[N0<gJgP3Ef]h0cM[N0=GJgP3=f]h0
eM[N0<gJgP3Ef]h000000=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0
g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d000000=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0
g/;50=Fn_@3N`/D0eKjm0=k2a@000000g[Zm0=Fj_@3N^[d0eKZm0=jj_@3E^[d0g[Zm0=Fj_@000000
cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0
cJjm0000000A0;BI_@1F0000002;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0
Rgg508=i_@2;OLD0PgVm08]ma@000000DV75059Ma@1BHLD0DUg5059Qa@1BGLD0DV75059Ma@1BHLD0
DUg5059Qa@1BGLD0000004YYc@1:JMD0BVW=04YYe@1:JLd0BVWE04YYc@1:JMD0BVW=04YYe@1:JLd0
BVWE04YYc@000000LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0
LiWV07^IiP1cVNH0NiWV0000003=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0
cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H000000<g>iP35c]h0cLkV0<G>gP00000<
0?ooo`030000003oool0oooo00X0oooo000W0?ooo`030000003oool0oooo00P0oooo01X0000007>b
o`1Z[_H0Lk;o06Z^mP1c/_l0JZkf07>bo`1Z[_H0Lk;o06Z^mP1c/_l0JZkf07>bo`000000Y;k^09bn
kP2T_^h0W;k^0:BnkP2L_^h0Y;k^09bnkP2T_^h0W;k^0:BnkP80000000d0]<KV0;g:iP2da^H0_L[V
0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0080000005D0eM[N0<gFgP3Ef]h0cMKN
0=GJgP3=e]h0eM[N0000003Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=
0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0000003N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm
0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@000000eKZm0=Ff]03E^[d0eKJd0=Fj_@3E][@00000
0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd
0<f^_@3=Z[@000000;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm
0;BE]02dVKd0]9Fd0;BI_@2dUK@0000000l0PgVm00l00000059M_@1BHLD0DUfm059Qa@1BGKd0DV75
059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M_@000000301:JLd0:0000000LiWV07>EiP1cVNH0LiGV
07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV00000035b]h0aLkN
0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N
0<G>gP35b]h0aLkN00000035c]h2000000030<G>gP000000oooo00X0oooo00<000000?ooo`3oool0
2P3oool002P0oooo0P0000090?ooo`0G0000001c/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f
07>bo`1Z/_H0Lk;o06ZbmP1c/_l000000:C2kP2T_^h0Y<;^0:BnkP2T`^h0Y;k^0:C2kP2T_^h00P00
00004@2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV
0;g:iP2da^H0_L[V008000000440cM[N0=GJgP3=f]h0eM[N0<gJgP000000g/k50=k>c@3Nc/D0g/k=
0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@000000eKjm
0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d00000
0=Fj_@3N^[d0eKZm0=jj_@3E^[d000000<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd
0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0000000B0;BI_@1F00000023NKd0Rgg5
08=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@2;OLD0PgVm08]ma@23NKd0Rgg508=i_@000000DUg5
059Qa@1BGLD0DV75059Ma@1BHLD0DUg5059Qa@1BGLD0DV75059Ma@1BHLD0000004YYe@1:JLd0BVWE
04YYc@1:JMD0BVW=04YYe@1:JLd0BVWE04YYc@1:JMD0BVW=04YYe@000000NiWV07>IiP1kVNH0LiWV
07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV00000035c]h0cLkV
0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN
0<g>iP35c]h0cLkV0000003=d^H0aM;V0<gBiP00000;0?ooo`030000003oool0oooo00X0oooo000Z
0?ooo`030000003oool0oooo00L0oooo0P0000004P1Z[_H0Lk;o06Z^mP1c/_l0JZkf07>bo`1Z[_H0
Lk;o06Z^mP1c/_l0JZkf07>bo`000000Y;k^09bnkP2T_^h0W;k^0:BnkP80000001D0]<KV0;g:iP2d
a^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2m
b^H0]<KV0;g:iP2da^H00P000000D@3Ef]h0cMKN0000003Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3N
b/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50000003E_[d0g/;50=Fn_@3N
`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003E
][@0eKZm0=Ff]0000000cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=
[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@000000]9Vm0;BE]02dVKd0]9Fd0;BI_@2d
UK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@0B0000000=059M_@1B
HLD0DUfm059Qa@1BGKd0DV75059M_@1BHLD0DUfm059Qa@1BGKd0DV750000000=04YYc@0S0000001c
VNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1c
UNH000000<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35
c]h0aL[N0<G>gP35b]h00`0000001@35c]h0aM;V0<G>gP35d^H0000000X0oooo00<000000?ooo`3o
ool02P3oool002/0oooo00<000000?ooo`3oool0203oool70000000906ZbmP1c/_l0J[;f07>bo`1Z
/_H0Lk;o0000002T`^h0Y;k^0080000001T0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2m
b^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2d
a^H0_L[V0080000003h0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3N
c/d0g/k50=k>c@3Nc/D0g/k=0=k>a@000000eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E
_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003E^[d0g[Zm0000003=
[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=
Z[@0cJjm0<fZ]03=[[d0cJZd000000l0]9Vm0`0000004@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2D
PKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@000000LgW500<0000004<0DUg5059Qa@1B
GLD0DV75059Ma@1BHLD0DUg5059Qa@1BGLD0000004YYc@1:JMD0BVW=04YYe@1:JLd0BVWE04YYc@1:
JMD0BVW=04YYe@1:JLd0BVWE04YYc@000000LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV07^IiP1c
VNH0NiWV07>IiP1kVNH0LiWV07^IiP1cVNH0NiWV07>IiP000000aLkN0<g>iP35c]h0cLkV0<G>gP3=
c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h000000<GBiP3=d^H0aM;V00000035
d^H0cM;V0<GBiP3=d^H0000000T0oooo00<000000?ooo`3oool02P3oool002`0oooo00<000000?oo
o`3oool0203oool00`000000FZ[f05ZZmP0305ZZmPT0000006P0]<KV0;g:iP2da^H0_L[V0;C6iP2m
b^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2d
a^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H000000<G6e@000000g/[50=k>c@3Nb/D0g/k=0=k:a@3N
c/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0000003N`/D0eKjm0=k2a@3E
_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N
`/D000000=Fj_@000000cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=
Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d000000;BI_@2dUK@0]9Vm0;BE]02d
VKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@80000001L0[9Fm0;BI_@2/UKd0000008^1
_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m0000
001cMLD0LgW=07=ea@0300000008059M_@1BHLD0DUfm059Qa@1BGKd0DV75059M_@00000=04YYc@0O
0000001cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1cUNH0LiWV
07>EiP1cVNH000000<G>gP35b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h0aLkN0<G:gP35c]h0aL[N
0<G>gP020000000:0<G>gP35d^H0aLkN0<GBiP35c]h000000<G>gP35d^H0aLkN000000X0oooo00<0
00000?ooo`3oool02@3oool002d0oooo0P0000090?ooo`80000002D0HZ[o05ZZmP1RZ_l0FZ[f06:Z
o`1JZ_H0HZ[o05ZZmP1RZ_l0FZ[f06:Zo`000000_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6
iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:
iP2da^H0_L[V0080000000<0aL[E0<g:gP35b]D00P000000?03Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>
a@3Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d000000=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2
a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50000
003=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^
_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm000000/0]9Vm01d000000;BIa@2/VKd0]9W5
0:bI_@2dVLD0000009B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m
09B5a@2DPKd0U8G50000001kNLd0LgW507]ic@1cNLD0NgW=07=ia@020000000`059Qa@1BGLD0DV75
059Ma@1BHLD0000004YYe@1:JLd0BVWE04YYc@1:JMD0BVW=04YYe@1:JLd0BVWE04YYc@1:JMD0BVW=
04YYe@1:JLd0000007>IiP1kVNH0LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV07^IiP1cVNH0NiWV
07>IiP1kVNH0LiWV07^IiP000000cLkV0<G>gP3=c^H0aLkN0<g>iP35c]h0cLkV0<G>gP3=c^H0aLkN
0P0000003@3=d^H0aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V00000035d^H0cM;V0<GBiP000000
2@3oool00`000000oooo0?ooo`090?ooo`00;`3oool00`000000oooo0?ooo`080?ooo`030000001J
Z_H0FZ[f00T0FZ[f0P000000H`2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0
]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H000000<g:gP35a]D0
cL[N0<G6e@3=b]h0aLKE0<g:gP000000g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D0g/k=0=k:a@3Nc/d0
g/[50=k>c@000000g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0
g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D000000=Fja@000000cJjm0<fZ]03=[[d0
cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0
cJjm0<fZ]03=[[d0cJZd0000002dUK@0]9Vm0;BE]02dVKd0]9Fd0;BI_@2dUK@0]9Vm0;BE]0020000
000O0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0000002;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0
U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0000007=ic@1cMLD0LgW=07=ea@1cNLd0
LgG507=ic@0300000004059Qa@1BGKd0DV75000000d0BVW=02`0000007>IiP1cUNH0LiWV07>EiP1c
VNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH000000<G>gP35
b]h0aLkN0<G:gP35c]h0aL[N0<G>gP35b]h000000<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35
d^H0aLkN0<GBiP000000aM;V0<G>gP35d^H0aLkN000000P0oooo00<000000?ooo`3oool02@3oool0
0300oooo00<000000?ooo`3oool0203oool0J0000000HZ[o05ZZmP1RZ_l0FZ[f06:Zo`1JZ_H0HZ[o
05ZZmP1RZ_l0FZ[f06:Zo`1JZ_H000000;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV
0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H000000<g:gP35b]D0cL[N
0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0000003Nc/d0g/k50=k>c@3Nc/D0g/k=0=k>a@3Nc/d0g/k5
0=k>c@000000g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;5
0=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D000000=Fna@3E^/D000000<fZ]03=[[d0cJZd
0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm
0<fZ]03=[[d0cJZd0<f^_@3=Z[@000001P2dVKd20000000T0:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W5
0:bI_@2dVLD0[9Vm0000002DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m
09B5a@2DPKd0U8G509B1_@000000LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW507]ic@1cNLD0NgW=
100000009P1:JLd0BVWE04YYc@1:JMD0BVW=04YYe@1:JLd0BVWE04YYc@1:JMD0BVW=04YYe@1:JLd0
BVWE0000001kVNH0LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0
LiWV07^IiP1cVNH0NiWV0000003=c^H0aLkN0<g>iP35c]h0cLkV0P0000004@35d^H0cM;V0<GBiP3=
d^H0aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V0<gBiP000000cM;V0<GBiP3=d^H0000000P0oooo
00<000000?ooo`3oool02@3oool00340oooo00<000000?ooo`3oool0203oool2000000/0FZ[f0180
00000;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0
]<KV0;g:iP2da^H20000000;0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=
b]h00P000000J03Nb/D0g/k=0=k:a@3Nc/d0g/[50=k>c@3Nb/D000000=Fn_@3N`/D0eKjm0=k2a@3E
_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N
`/D0eKjm0000003=^[d0eK[50<fj_@000000cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=
Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0000002d
VKd0]9Fd0;BI_@2dUK@0]9Vm0000002dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/
UKd0]9Vm0000002DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2D
PLD0Rh6m09B1a@000000LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@1c
MLD0LgW=1000000;04YYc@0/0000001cUNH0LiWV07>EiP1cVNH0LiGV07>IiP1cUNH0LiWV07>EiP1c
VNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0LiGV00000035b]h0aLkN0<G:gP35c]h000000<GBiP35
c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V00000035
d^H0aLkN0<GBiP0000070?ooo`030000003oool0oooo00T0oooo000b0?ooo`030000003oool0oooo
00T0oooo06D0000005ZZmP1RZ_l0FZ[f06:Zo`1JZ_H0HZ[o05ZZmP1RZ_l0FZ[f06:Zo`1JZ_H00000
0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV
00000035b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N
0<G:e@000000g/k50=k>c@3Nc/D0g/k=0=k>a@000000eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm
0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d00000
0=Fja@3E_/D0eK[50=Fna@000000cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd
0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd000000030;BI_@80
000002h0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0000009B1
_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5
a@000000NgW=07=ia@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW507]ic@1cNLD0NgW=0000
001kRMh0LhGE10000000701:JMD0BVW=04YYe@1:JLd0BVWE04YYc@1:JMD0BVW=0000001cVNH0NiWV
07>IiP1kVNH0LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV07^IiP1cVNH0NiWV07>IiP1kVNH0LiWV
00000035c]h20000000F0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0
aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H000000<gBiP35d^H0cM;V000000H0oooo00<000000?ooo`3o
ool02@3oool003<0oooo0P00000:0?ooo`030000001JZ_H0FZ[f00T0FZ[f0P000000EP2mb^H0]<KV
0;g:iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0;C6iP000000aLKE0<g:gP35a]D0cL[N
0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@000000g/[5
0=k>c@3Nb/D000000=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;5
0=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003=^[d0eK[50<fj_@3E^/D0cKZm
0=Fja@000000cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm
0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0`000000=02/UKd0]9Vm0:bE_@2dVKd0
[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@000000Rh6m09B1a@2;PKd0
U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0000007=ea@1cNLd0
LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0000007>9e@1cQMD0
LhWE07>5e@1cRMD4000000D0BVW=00h0000007>EiP1cVNH0LiGV07>IiP1cUNH0LiWV07>EiP1cVNH0
LiGV07>IiP1cUNH0LiWV07>EiPD0000001P0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35
c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN00000035c]h0aM;V0<G>gP00
00060?ooo`030000003oool0oooo00T0oooo000e0?ooo`030000003oool0oooo00P0oooo0P000000
5@1RZ_l0FZ[f06:Zo`1JZ_H0HZ[o05ZZmP1RZ_l0FZ[f06:Zo`1JZ_H0HZ[o0000002mb^H0]<KV0;g:
iP2da^H0_L[V0;C6iP2mb^H0]<KV0;g:iP020000000C0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:
e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@020000001`0=k>
c@000000g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn
_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@000000eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50000
003=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ
]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d000000;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BI
a@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0000009B5a@2DPKd0U8G509B1
_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G50000001kNLd0LgW507]i
c@1cNLD0NgW=07=ia@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW50000001cQMD0NhWN07>5
e@1kRMh0LhGE07^9gP1cQMD0NhWN07>5e@@0000000L0BVWE0000001kVNH0LiWV07^IiP1cVNH0NiWV
00P0000001h0U:kV09B^kP2D[^H0U:k^09B^iP000000aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V
0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V00000035d^H0cM;V
0<GBiP0000060?ooo`030000003oool0oooo00P0oooo000f0?ooo`030000003oool0oooo00T0oooo
00<0000005ZZmP1JZ_H02@1JZ_H0E0000000_L[V0;C6iP2mb^H0]<KV0;g:iP2da^H0_L[V0000003=
b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35
a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0000003N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N
`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@00
0000eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@000000cJZd0<f^_@3=Z[@0cJjm0<fZ]03=
[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@80
000003h0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI
_@2/UKd0]9Vm0:bE_@2dVKd0000009B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1
_@2DPLD0Rh6m09B1a@2;PKd0U8750000001cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=i
c@1cMLD0LgW=07=ea@1cNLd0LgG50000001cQMD0LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5
e@1cRMD0LhGE07>9e@L000000280U:k^08^ZiP2D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V
09B^kP2;Z^H0U:k^08^ZiP000000aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V
0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0`0000000`2mf^H000000?ooo`040?oo
o`030000003oool0oooo00P0oooo000g0?ooo`030000003oool0oooo00T0oooo10000000201JZ_H0
HZ[o05ZZmP1RZ_l0FZ[f06:Zo`1JZ_H0HZ[o0P000000D@2da^H0_L[V0;C6iP2mb^H000000<g:gP35
b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=
b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h000000<g:e@000000g/;50=Fn_@3N`/D0eKjm0=k2a@3E
_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50000003E
_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@000000cJjm0<fZ]03=[[d0cJZd0<f^_@3=
Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm00000002
0<F^a@1V0000002/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/
VKd0]9W50:bI_@2dVLD0[9Vm0000002DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2D
QLD0U86m09B5a@2DPKd0U8G509B1_@000000LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW507]ic@1c
NLD0NgW=07=ia@1kNLd0LgW507]ic@000000NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN07>5e@1k
RMh0LhGE07^9gP1cQMD0NhWN07>5e@1kRMh0000009B^kP2D[^H0U:k^09B^iP2D[^h0U:kV09B^kP2D
[^H0U:k^09B^iP2D[^h0U:kV09B^kP2D[^H0U:k^09B^iP2D[^h0U:kV00000035d^H0cM;V0<GBiP3=
d^H0aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V0P000000
1@35g^h0aM[^0<GNkP35f^h0000000@0oooo00<000000?ooo`3oool0203oool003P0oooo0P00000:
0?ooo`030000001JYOH0DZ7f00P000000P1JZ_H0]`000000]<KV000000000000cL[N0<G6e@3=b]h0
aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0
cL[N0<G6e@3=b]h0aLKE0<g:gP000000cLKE0<g6c@3=a]D000000=k2a@3E_[d0g/;50=Fn_@3N`/D0
eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D000000=Fja@3=^[d0
eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D000000<f^_@3=Z[@0cJjm0<fZ]03=[[d0
cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d000000<F^a@35Z[d0
aJk50000002dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0
[9Fm0;BI_@2/UKd0]9Vm0000002DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0
U87508^1_@2DPLD0Rh6m09B1a@2;PKd0000007=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0
LgG507=ic@1cMLD0LgW=07=ea@1cNLd0000007>9e@1cQMD0LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0
LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0000008^ZiP2D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h0
Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP000000aLkN0<GBiP35c]h0
aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>gP35d^H0aLkN0<GBiP02000000070;gJiP35f^h0
00000<GJkP2mf^H0aM[^000000040?ooo`030000003oool0oooo00P0oooo000j0?ooo`030000003o
ool0oooo00P0oooo00X0000005ZUo`1BYOH0FZGo05:UmP1JYOl0DZGf05ZUo`1BYOH0FZGo10000000
C@35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:
e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP000000cL[E0<g6e@3=b]D0cLKE0<g:
e@000000g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn
_@3N`/D000000=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50000
003=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ
]03=[[d0000000D0aJk506d000000;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI
_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0000009B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5
a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G50000001kNLd0LgW507]ic@1cNLD0NgW=07=i
a@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW50000001cQMD0NhWN07>5e@1kRMh0LhGE07^9
gP1cQMD0NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN07>5e@000000U:kV09B^kP2D[^H0U:k^09B^
iP2D[^h0U:kV09B^kP2D[^H0U:k^09B^iP2D[^h0U:kV09B^kP2D[^H0U:k^09B^iP2D[^h000000<gB
iP35d^H0cM;V0<GBiP3=d^H0aM;V0<gBiP35d^H0cM;V0<GBiP3=d^H0aM;V0<gBiP000000aMk^0<GJ
kP35g^h0aM[^0<GNkP000000aMk^0<GJkP35g^h0000000<0oooo00<000000?ooo`3oool0203oool0
03/0oooo00<000000?ooo`3oool0203oool20000000Q05:QmP1JYOH0DZ7f05ZUmP1BXOH0FZGf05:Q
mP1JYOH0DZ7f05ZUmP1BXOH000000<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:
gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0080000009X0cLKE0<g6
c@3=a]D0cLK=0<g6e@3=a/d0cLKE0000003N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2
a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj
_@3E^/D0cKZm0=Fja@3=^[d000000<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^
_@3=Z[@0cJjm0<fZ]03=[[d000000<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0000002/UKd0]9Vm0:bE
_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0000
002;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1
_@000000LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=i
c@1cMLD0000007>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5
e@1cRMD0LhGE0000002;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^
kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^kP000000aM;V0<G>gP35d^H0aLkN0<GBiP35c]h0aM;V0<G>
gP35d^H0aLkN0P0000003035f^h0_M[V0<GJkP2mf^H0aM[^0;gJiP35f^h000000<GJkP2mf^H0aM[^
00000080oooo00<000000?ooo`3oool0203oool003`0oooo00<000000?ooo`3oool02@3oool0EP00
0000DZGf05ZUo`1BYOH0FZGo05:UmP1JYOl0DZGf05ZUo`1BYOH0FZGo05:UmP000000aL[E0<g:gP35
b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=
b]h0aL[E0000003=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D000000=k2a@3E
_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003E^/D0eKk50=Fja@3E
_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50000003=[[d0cJZd0<f^_@3=
Z[@0cJjm0<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d000001`35[/D0GP000000]9W50:bI
_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BI
a@000000U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1
_@2DQLD0U86m0000001cNLD0NgW=07=ia@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW507]i
c@1cNLD0NgW=0000001kRMh0LhGE07^9gP1cQMD0NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN07>5
e@1kRMh0LhGE07^9gP000000U:k^09B^iP2D[^h0U:kV09B^kP2D[^H0U:k^09B^iP2D[^h0U:kV09B^
kP2D[^H0U:k^09B^iP2D[^h0U:kV09B^kP2D[^H0U:k^0000003=d^H0aM;V0<gBiP35d^H0cM;V0<GB
iP3=d^H20000000A0<GJkP35g^h0aM[^0<GNkP35f^h0aMk^0<GJkP35g^h0aM[^00000035f^h0aMk^
0<GJkP35g^h000000?ooo`0000002P3oool003d0oooo0P00000:0?ooo`2h0000001BXOH0FZGf05:Q
mP1JYOH0DZ7f05ZUmP1BXOH0FZGf05:QmP1JYOH0DZ7f00000035a]D0cL[N0<G6e@3=b]h0aLKE0<g:
gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0000003=a/d0cLKE0<g6
c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@000000g/;50=Fn_@3N`/D0eKjm0=k2
a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj
_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D000000<f^_@3=Z[@0cJjm0<fZ]03=[[d0cJZd0<f^
_@3=Z[@0cJjm0<fZ]03=[[d000000<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50000
002dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI
_@2/UKd0]9Vm0000002DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1
_@2DPLD0Rh6m09B1a@000000LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=i
c@1cMLD0LgW=07=ea@1cNLd0000007>9e@1cQMD0LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5
e@1cRMD0LhGE07>9e@1cQMD0LhWE0000002D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^
kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h000000<GBiP35c]h0aM;V0<G>
gP8000000140aM[^0;gJiP35f^h0_M[V0<GJkP2mf^H0aM[^0;gJiP35f^h0_M[V0<GJkP2mf^H00000
0;gJiP35f^h0_M[V0<GJkP02000000X0oooo000o0?ooo`030000003oool0oooo00P0oooo05800000
05:UmP1JYOl0DZGf05ZUo`1BYOH0FZGo05:UmP1JYOl0DZGf05ZUo`1BYOH000000<G:e@3=b]h0aL[E
0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0cL[N0<G:e@3=b]h0aL[E0000003=a]D0cL[E
0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D000000=Fn_@3N`/D0eKjm
0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0000003E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk5
0=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50000003=Z[@0cJjm0<fZ]03=[[d0cJZd
0<f^_@3=Z[@0cJjm0<fZ]03=[[d000002P35[/D0F@000000[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0
[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@000000U86m09B5a@2DPKd0
U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G50000001kNLd0
LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW50000001cQMD0
NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN07>5e@1kRMh0
000009B^kP2D[^H0U:k^09B^iP2D[^h0U:kV09B^kP2D[^H0U:k^09B^iP2D[^h0U:kV09B^kP2D[^H0
U:k^09B^iP2D[^h0U:kV09B^kP000000cM;V0080000001H0aM[^0<GNkP35f^h0aMk^0<GJkP35g^h0
aM[^0<GNkP35f^h0aMk^0<GJkP35g^h0aM[^0<GNkP35f^h000000<GJkP35g^h0aM[^0000003oool0
00002@3oool00400oooo00<000000?ooo`3oool0203oool0/P000000DZ7f05ZUmP1BXOH0FZGf05:Q
mP1JYOH0DZ7f05ZUmP1BXOH0FZGf05:QmP000000aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:
gP35a]D0cL[N0<G6e@3=b]h0aLKE0000003=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6
c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@000000eKjm0=k2a@3E_[d0g/;50=Fn_@3N`/D0eKjm0=k2
a@000000eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj
_@3E^/D0cKZm0=Fja@3=^[d000000<fZ]03=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d000000<F^
a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0000002/UKd0]9Vm0:bE
_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0000009B1
a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1
_@000000LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=i
c@1cMLD0000007>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5
e@1cRMD0LhGE07>9e@000000U:k^08^ZiP2D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^
kP2;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^kP2;Z^H20000000F0<GJkP2mf^H0aM[^0;gJiP35f^h0_M[V
0<GJkP2mf^H0aM[^0;gJiP35f^h0_M[V0<GJkP2mf^H0aM[^0;gJiP35f^h000000<GJkP2mf^H0aM[^
0;gJiP8000002@3oool00440oooo00<000000?ooo`3oool0203oool20000000:05ZUo`1BYOH0FZGo
05:UmP1JYOl0DZGf05ZUo`1BYOH0FZGo05:UmP8000000400cL[N0<G:e@3=b]h0aL[E0<g:gP35b]D0
cL[N0<G:e@3=b]h0aL[E0000003=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0
cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0000003E_[d0g/;50=Fn_@3N`/D0eKjm0=k2a@000000
eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0
eK[50=Fna@3E^/D0eKk50000003=[[d0cJZd0<f^_@3=Z[@0cJjm0<fZ]03=[[d000003@35[/D0DP00
0000]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2d
VLD0[9Vm0;BIa@000000U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2D
PKd0U8G509B1_@2DQLD0000007]ic@1cNLD0NgW=07=ia@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1k
NLd0LgW507]ic@1cNLD0NgW=0000001kRMh0LhGE07^9gP1cQMD0NhWN07>5e@1kRMh0LhGE07^9gP1c
QMD0NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0000009B^iP2D[^h0U:kV09B^kP2D[^H0U:k^09B^iP2D
[^h0U:kV09B^kP2D[^H0U:k^09B^iP2D[^h400000080Y<kf01H000000<GNkP35f^h0aMk^0<GJkP35
g^h0aM[^0<GNkP35f^h0aMk^0<GJkP35g^h0aM[^0<GNkP35f^h0aMk^0<GJkP35g^h000000<GNkP35
f^h0aMk^0P0000090?ooo`00@P3oool2000000X0oooo0:P0000005ZUmP1BXOH0FZGf05:QmP1JYOH0
DZ7f05ZUmP1BXOH0FZGf05:QmP1JYOH000000<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0
00000<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0
cLK=0<g6e@3=a/d0cLKE0<g6c@000000eKjm0=k2a@3E_[d0g/;50000003E^/D0cKZm0=Fja@3=^[d0
eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0
cKZm0=Fja@000000cJjm0<fZ]03=[[d0cJZd0<f^_@000000aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0
aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@000000[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0
[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@000000Rh6m09B1a@2;PKd0
U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U8750000001cNLd0
LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@000000
LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0LhWE07>5e@1cRMD0
LhGE0000002;Z^H0U:k^08^ZiP2D[^h0Rj[V09B^kP2;Z^H0U:k^08^ZiP2D[^h3000000L0Y<kf0180
00000<GJkP2mf^H0aM[^0;gJiP35f^h0_M[V0<GJkP2mf^H0aM[^0;gJiP35f^h0_M[V0<GJkP2mf^H0
aM[^0;gJiP35f^h3000000030?ooo`000000oooo00P0oooo00140?ooo`030000003oool0oooo00P0
oooo0180000005ZUo`1BYOH0FZGo05:UmP1JYOl0DZGf05ZUo`1BYOH0FZGo05:UmP1JYOl000000<g:
gP35b]D0cL[N0<G:e@3=b]h20000000e0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE
0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0000003E_[d0g/;5
0000003E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[5
0=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D000000<fZ]03=[[d0cJZd0<f^_@0000004035
[/D0B`000000[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm
0;BIa@2/VKd0]9W50000002DQLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G5
09B1_@2DQLD0U86m09B5a@2DPKd0000007=ia@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW5
07]ic@1cNLD0NgW=07=ia@1kNLd0LgW50000001cQMD0NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN
07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN07>5e@1kRMh0LhGE0000002D[^H0U:k^09B^iP2D[^h0U:kV
09B^kP03000000/0Y<kf010000000<GNkP35f^h0aMk^0<GJkP35g^h0aM[^0<GNkP35f^h0aMk^0<GJ
kP35g^h0aM[^0<GNkP35f^h0aMk^0`0000020?ooo`030000003oool0oooo00L0oooo00150?ooo`03
0000003oool0oooo00P0oooo10000000V`1BXOH0FZGf05:QmP1JYOH0DZ7f05ZUmP1BXOH0FZGf0000
003=b]h0aLKE0<g:gP000000cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6
e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d000000=Fn
_@000000cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fj
a@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@000000cJZd0<f^_@000000aJk50<FZ
_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^
a@000000]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE
_@2dVKd0[9Fm0;BI_@000000U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1
a@2;PKd0U87508^1_@2DPLD0000007=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=e
a@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@000000LhGE07>9e@1cQMD0LhWE07>5e@1cRMD0LhGE07>9
e@1cQMD0LhWE07>5e@1cRMD0LhGE07>9e@1cQMD0LhWE0000002D[^h0Rj[V00@000003`2Tc_H03@00
0000aM[^0;gJiP35f^h0_M[V0<GJkP2mf^H0aM[^0;gJiP35f^h0_M[V0<GJkP2mf^H00P0000020?oo
o`040000003oool0oooo000000T0oooo00160?ooo`030000003oool0oooo00P0oooo00<0000005ZU
mP1JXOH01`000000>`1BYOH0FZGo0000003=b]h000000<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:
e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6
e@3=b]D0cLKE0<g:e@3=a]D000000=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fn
a@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50000
003=[[d0000001<0aJk5040000000;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI
_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@000000U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m09B5
a@2DPKd0U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G50000001kNLd0LgW507]ic@1cNLD0NgW=07=i
a@1kNLd0LgW507]ic@1cNLD0NgW=07=ia@1kNLd0LgW507]ic@000000NhWN07>5e@1kRMh0LhGE07^9
gP1cQMD0NhWN07>5e@1kRMh0LhGE07^9gP1cQMD0NhWN1P00000D0:C>mP0900000035g^h0aM[^0<GN
kP35f^h0aMk^0<GJkP35g^h0aM[^00<000001@3oool00`000000oooo000000090?ooo`00A`3oool2
000000T0oooo00T0000005ZUmP1JXOH0FZGf05ZQmP1JYOH0FZ7f05ZUmP1JXOH0100000006@3=a]D0
cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0
cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D00P000000I`35a]D000000<fj_@3E^/D0
cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0
eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[500000035[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0
aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0000002/UKd0
]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0
[9Fm0000002;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0
U87508^1_@000000LgG507=ic@1cMLD0LgW=07=ea@1cNLd0LgG507=ic@1cMLD0LgW=07=ea@1cNLd0
LgG507=ic@1cMLD0LgW=0000001cRMD0LhGE07>9e@1cQMD0200000001P2D^_H0Rk[^09BjmP2;^^h0
U;[f000001D0Y<kf00H000000<GJkP2mf^H0aM[^0;gJiP35f^h2000000T0oooo00<000000?ooo`00
0000203oool004T0oooo00<000000?ooo`3oool01`3oool20000001105ZQmP1JYOH0FZ7f05ZUmP1J
XOH0FZGf05ZQmP1JYOH0FZ7f05ZUmP000000cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=
b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@00
0000cL[N0<G6e@3=b]h0aLKE0000003E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E
_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D000000<fne@00
00004`35[/D0;0000000]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/
VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@000000U8G509B1_@2DQLD0U86m09B5a@2DPKd0U8G509B1_@2D
QLD0U86m09B5a@2DPKd0U8G509B1_@2DQLD0U86m0000001cNLD0NgW=07=ia@1kNLd0LgW507]ic@1c
NLD0NgW=300000003`2D^^h0U;[f09BjkP2D^_H0U;[^09BjmP2D^^h0U;[f09BjkP2D^_H0U;[^09Bj
mP2D^^h0U;[f0000000E0:C>mP0300000035g^h0aM[^00800000303oool2000000P0oooo001:0?oo
o`030000003oool0oooo00P0oooo0200000005ZQmP1JYOH0FZ7f05ZUmP1JXOH0FZGf05ZQmP1JYOH0
FZ7f05ZUmP000000cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0
cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@8000000580cLKN0<G6e@3=a]h0aLKE0<g6gP00
0000eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E
^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D000000<fnc@35_/d000000<FZ_@35[/D0aJZm0<F^a@35
Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35
[/D000000;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/
UKd0]9Vm0:bE_@000000Rh6m09B1a@2;PKd0U87508^1_@2DPLD0Rh6m09B1a@2;PKd0U87508^1_@2D
PLD0Rh6m30000000702;Z^h0RjGV08^ZkP2;YNH0Rj[^08^UiP2;Z^h0RjGV0000002;^^h0U;[f08^j
kP2D^_H0Rk[^09BjmP2;^^h0U;[f08^jkP2D^_H0Rk[^09BjmP2;^^h0U;[f08^jkP2D^_H0Rk[^09Bj
mP00000D0:C>mP<000003P3oool2000000P0oooo001;0?ooo`030000003oool0oooo00P0oooo00/0
000005ZQmP1JYOH0FZ7f05ZUmP1JXOH0FZGf05ZQmP1JYOH0FZ7f05ZUmP020000000]0<g6e@3=b]D0
cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:e@3=a]D0
00000<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h000000=Fna@3E^/D0eKk50=Fja@3E_/D0
eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50080000000D0
cKkE0<Fnc@3=_]D0aKk=0000000C0<F^a@0G0000002/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0
]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0000002DPKd0U8G509B1_@2DQLD0
20000000:@2LZ]h0W:kV09bZgP2L[^H0000008^ZkP2;Z^H0Rj[^08^ZiP2;Z^h0Rj[V08^ZkP2;Z^H0
Rj[^08^ZiP2;Z^h0Rj[V08^ZkP2;Z^H0Rj[^08^ZiP000000U;[^09BjmP2D^^h0U;[f09BjkP2D^_H0
U;[^09BjmP2D^^h0U;[f09BjkP2D^_H0U;[^09BjmP2D^^h0U;[f09BjkP2D^_H0000000l0Y<kf1000
000A0?ooo`800000203oool004`0oooo0P0000090?ooo`1T0000001JXOH0FZGf05ZQmP1JYOH0FZ7f
05ZUmP1JXOH0FZGf05ZQmP1JYOH0FZ7f0000003=a/d0cLKE0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE
0<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE0<g6c@000000aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE
0<g6gP35a]D0cLKN0000003E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[5
0<fj_@3E^/D0cKZm0=Fja@3=^[d000000<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=00000035[/D0aJZm
0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk5
0<FZ_@35[/D0aJZm0000002/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd0[9Fm
0;BI_@2/UKd0]9Vm0:bE_@H000000380W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0
U:[N09bZiP2DZ]h0000008^UiP2;Z^h0RjGV08^ZkP2;YNH0Rj[^08^UiP2;Z^h0RjGV08^ZkP2;YNH0
Rj[^08^UiP2;Z^h0RjGV08^ZkP2;YNH0000008^jkP2D^_H0Rk[^09BjmP2;^^h0U;[f08^jkP2D^_H0
Rk[^09BjmP2;^^h0U;[f08^jkP2D^_H0Rk[^09BjmP2;^^h0U;[f000000/0Y<kf0`00000A0?ooo`@0
00002P3oool004h0oooo00<000000?ooo`3oool01`3oool06@000000FZ7f05ZUmP1JXOH0FZGf05ZQ
mP1JYOH0FZ7f05ZUmP1JXOH0FZGf05ZQmP000000cLKE0<g:e@3=a]D0cL[E0<g6e@3=b]D0cLKE0<g:
e@3=a]D0cL[E0<g6e@3=b]D00P0000009@35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6
e@3=b]h0aLKE0<g:gP000000eKk50=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E_/D0eK[50=Fn
a@3E^/D0eKk50=Fja@000000aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@0000004`35[/D0
3P000000]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50P00
0000>@2/]]h0];[V0000002L[^H0W:[N09b^iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h0W:kV
09bZgP2L[^H0W:[N09b^iP2LZ]h0000008^ZiP2;Z^h0Rj[V08^ZkP2;Z^H0Rj[^08^ZiP2;Z^h0Rj[V
08^ZkP2;Z^H0Rj[^08^ZiP2;Z^h0Rj[V08^ZkP2;Z^H0000009BjkP2D^_H0U;[^09BjmP2D^^h0U;[f
09BjkP2D^_H0U;[^09BjmP2D^^h0U;[f09BjkP2D^_H0U;[^09BjmP2D^^h0U;[f000000070:C>mP<0
0000403oool4000000h0oooo001?0?ooo`030000003oool0oooo00L0oooo0P000000F`1JYOH0FZ7f
05ZUmP1JXOH0FZGf05ZQmP1JYOH0FZ7f05ZUmP1JXOH000000<g6c@3=a]D0cLK=0<g6e@3=a/d0cLKE
0<g6c@3=a]D0cLK=0<g6e@000000cLKN0<G6e@3=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE
0<g6gP35a]D0cLKN0<G6e@000000cKZm0=Fja@3=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[5
0<fj_@3E^/D0cKZm00000035_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@000000aJZm
0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk5
0<FZ_@35[/D0aJZm0000002/UKd0]9Vm0:bE_@2dVKd0[9Fm0;BI_@2/UKd0]9Vm0:bE_@2dVKd00`00
0000?02d^]h0[;KN0;BjgP2/]]h0000009BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0U:[N
09bZiP2DZ]h0W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0000008^UiP2;Z^h0RjGV08^ZkP2;YNH0Rj[^
08^UiP2;Z^h0RjGV08^ZkP2;YNH0Rj[^08^UiP2;Z^h0RjGV08^ZkP000000U;[f08^jkP2D^_H0Rk[^
09BjmP2;^^h0U;[f08^jkP2D^_H0Rk[^09BjmP2;^^h0U;[f08^jkP2D^_H0Rk[^09BjmP2;^^h0U;[f
00000080Y<kf1000000?0?ooo`@000004P3oool00500oooo00<000000?ooo`3oool0203oool04`00
0000FZGf05ZQmP1JYOH0FZ7f05ZUmP1JXOH0FZGf05ZQmP1JYOH0FZ7f0000003=a]D0cL[E0<g6e@3=
b]D0cLKE0<g:e@3=a]D00P000000:@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35
a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D000000=Fja@3E_/D0eK[50=Fna@3E^/D0eKk50=Fja@3E
_/D0eK[50=Fna@3E^/D000000<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35
_/d0000001<0aJk500P000000:bI_@2dVLD0[9Vm0;BIa@2/VKd0]9W50:bI_@80000003h0];[V0:bf
gP2d^^H0[;KN0;BjiP2/]]h0];[V0:bfgP000000W:[N09b^iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^
iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h0W:kV09bZgP000000Rj[V08^ZkP2;Z^H0Rj[^08^Z
iP2;Z^h0Rj[V08^ZkP2;Z^H0Rj[^08^ZiP2;Z^h0Rj[V08^ZkP2;Z^H0Rj[^0000002D^_H0U;[^09Bj
mP2D^^h0U;[f09BjkP2D^_H0U;[^09BjmP2D^^h0U;[f09BjkP2D^_H0U;[^09BjmP2D^^h0U;[f09Bj
kP<000003`3oool4000001H0oooo001A0?ooo`8000002@3oool40000000705ZQmP1JYOH0FZ7f05ZU
mP1JXOH0FZGf05ZQmP02000000180<g6e@3=a/d0cLKE0<g6c@000000aLKE0<g6gP35a]D0cLKN0<G6
e@3=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0000
003=^[d0eK[50<fj_@3E^/D0cKZm0=Fja@3=^[d0eK[50<fj_@000000aKk=0<fnc@35_/d0cKk=0<Fn
c@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d000000<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^
a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@000000]9Vm0:bE
_@2dVKd0[9Fm0`000000?02/]]h0];[N0:bfgP2d^]h0[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN
0000002DZ]h0W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V
09BZgP2LZ^H0000008^ZkP2;YNH0Rj[^08^UiP2;Z^h0RjGV08^ZkP2;YNH0Rj[^08^UiP2;Z^h0RjGV
08^ZkP2;YNH0Rj[^08^UiP2;Z^h0000009BjmP2;^^h0U;[f08^jkP2D^_H0Rk[^09BjmP2;^^h0U;[f
08^jkP2D^_H0Rk[^09BjmP@000003P3oool4000001X0oooo001C0?ooo`030000003oool0oooo00L0
oooo00<0000006:UmP1JXOH01`000000101JXOH0FZGf0000003=b]D20000000]0<G6e@3=b]h0aLKE
0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N
0<G6e@3=b]h0aLKE0000003E^/D0eKk50=Fja@3E_/D0eK[50=Fna@3E^/D000000<Fnc@3=_]D0aKk=
0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0000001<0aJk503l00000
0;BIa@00000000000:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V
0:bfgP2d^^H0[;KN0000002LZ]h0W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h0W:kV09bZgP2L[^H0W:[N
09b^iP2LZ]h0W:kV09bZgP2L[^H0000008^ZkP2;Z^H0Rj[^08^ZiP2;Z^h0Rj[V08^ZkP2;Z^H0Rj[^
08^ZiP2;Z^h0Rj[V08^ZkP2;Z^H0Rj[^08^ZiP2;Z^h0000009BjmP2D^^h0U;[f09BjkP2D^_H0U;[^
09BjmP2D^^h01000000>0?ooo`@000007P3oool005@0oooo00<000000?ooo`3oool01`3oool02@00
0000HZGf05ZQmP1RYOH0FZ7f06:UmP1JXOH0HZGf05ZQmP040000000M0<g6gP35a]D0cLKN0<G6e@3=
a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP35
a]D0cLKN0<G6e@3=a]h0aLKE0000003=^[d0eK[50<fj_@3E^/D00P0000008`35_/d0cKk=0<Fnc@3=
_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d000000<F^a@35
Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35
[/D0aJZm0080000003T0];[N0:bfgP2d^]h0[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN0;BjgP2/
]]h0];[N0:bfgP2d^]h0[;KN0;BjgP000000W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V09BZgP2L
Z^H0U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0U:[N09bZiP000000Rj[^08^UiP2;Z^h0RjGV08^ZkP2;
YNH0Rj[^08^UiP2;Z^h0RjGV08^ZkP2;YNH0Rj[^08^UiP2;Z^h0RjGV08^ZkP000000U;[f08^jkP2D
^_H01000000?0?ooo`<000008P3oool005D0oooo00<000000?ooo`3oool01`3oool20000000j05ZQ
mP1RYOH0FZ7f06:UmP1JXOH0HZGf05ZQmP1RYOH0FZ7f06:UmP000000cL[N0<G6e@3=b]h0aLKE0<g:
gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6
e@3=b]h0aLKE0<g:gP000000eKk50=Fja@3E_/D000000<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fn
e@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=00000100aJk50P000000
=P2ma^H000000:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bf
gP2d^^H0[;KN0;BjiP2/]]h0];[V0000002L[^H0W:[N09b^iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^
iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h0W:kV0000002;Z^h0Rj[V08^ZkP2;Z^H0Rj[^08^Z
iP2;Z^h0Rj[V08^ZkP2;Z^H0Rj[^08^ZiP2;Z^h0Rj[V08^ZkPD000003`3oool4000002D0oooo001F
0?ooo`8000002@3oool0B0000000FZ7f06:UmP1JXOH0HZGf05ZQmP1RYOH0FZ7f06:UmP1JXOH0HZGf
0000003=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE
0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP35a]D0cLKN0000003E^/D000000<fnc@35_/d0cKk=
0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=
0<fnc@35_/d000000<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0<F^a@35Z[d0aJk5
0<FZ_@80000003<0_LKV0;g6gP2ma^H0_LKN0000002/]]h0];[N0:bfgP2d^]h0[;KN0;BjgP2/]]h0
];[N0:bfgP2d^]h0[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN0;BjgP000000W:[V09BZgP2LZ^H0
U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0U:[N0000002;YNH0
Rj[^08^UiP2;Z^h0RjGV08^ZkP2;YNH0Rj[^08^UiP2;Z^h01@00000@0?ooo`@00000:@3oool005P0
oooo00<000000?ooo`3oool01`3oool02`000000FZ7f06:UmP1JXOH0HZGf05ZQmP1RYOH0FZ7f06:U
mP1JXOH0HZGf0080000001D0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6
e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D00P0000005P3=_]D0aKk=0<fn
e@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fn
c@3=_]D0aKk=0<fne@00000;0<F^a@80000002l0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H000000;Bj
iP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bf
gP2d^^H0[;KN0000002LZ]h0W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^
iP2LZ]h0W:kV09bZgP2L[^H0W:[N0000002;Z^H0Rj[^08^ZiP0600000140oooo1000000]0?ooo`00
F@3oool00`000000oooo0?ooo`070?ooo`0O0000001JXOH0HZGf05ZQmP1RYOH0FZ7f06:UmP1JXOH0
HZGf05ZQmP1RYOH0FZ7f00000035a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0
aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP020000000Q0;C:iP2mb^H000000<fnc@35_/d0
cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0
aKk=0<fnc@35_/d0cKk=00000035[/D0aJZm0<F^a@35Z[d0aJk50<FZ_@35[/D0aJZm0080000002d0
_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H000000;BjgP2/]]h0];[N0:bfgP2d^]h0
[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN0000002DZ]h0
W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0U:[N09bZiP2DZ]h0W:[V09BZgP2LZ^H0
1000000C0?ooo`@00000<@3oool005X0oooo00<000000?ooo`3oool01`3oool070000000FZ7f06:U
mP1JXOH0HZGf05ZQmP1RYOH0FZ7f06:UmP1JXOH0HZGf05ZQmP000000aLKE0<g:gP35a]D0cL[N0<G6
e@3=b]h0aLKE0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE0<g:gP35a]D20000000K0;g:kP2db^H0_L[^
0;C:iP2mb^h000000<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=
0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0000000H0aJk50P000000:`2ma^H0_L[V0;g6
iP2mb^H0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP000000];[V0:bfgP2d^^H0[;KN0;Bj
iP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0000009bZ
gP2L[^H0W:[N09b^iP2LZ]h0W:kV09bZgP2L[^H0W:[N09b^iP2LZ]h01000000C0?ooo`@00000=@3o
ool005/0oooo0P0000080?ooo`80000001L0HZGf05ZQmP1RYOH0FZ7f06:UmP1JXOH0HZGf05ZQmP1R
YOH0FZ7f00000035a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP35a]D0cLKN0<G6e@3=a]h0aLKE0<g6gP02
0000000Q0;C:iP2mb^H0]<[V0;g:iP2db^H0_L[V0;C:iP000000aKk=0<fnc@35_/d0cKk=0<Fnc@3=
_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35
_/d000000<FZ_@35[/D0aJZm0080000002T0_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2m
a^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H000000;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN0;BjgP2/
]]h0];[N0:bfgP2d^]h0[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN0000002DZ]h0W:[V09BZgP2L
Z^H0U:[N09bZiP04000001@0oooo0`00000i0?ooo`00G@3oool00`000000oooo0?ooo`070?ooo`0E
0000001RYOH0FZ7f06:UmP1JXOH0HZGf05ZQmP1RYOH0FZ7f06:UmP1JXOH000000<G6e@3=b]h0aLKE
0<g:gP35a]D0cL[N0<G6e@3=b]h0aLKE008000000240_L[^0;C:iP2mb^h0]<[V0;g:kP2db^H0_L[^
0;C:iP2mb^h0]<[V00000035_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=
0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE00000035[/D00P0000009`2ma^H0_L[V
0;g6iP2mb^H0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV
0;g:iP000000];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V
0:bfgP2d^^H0[;KN0;BjiP000000W:kV09bZgP04000001@0oooo1000000l0?ooo`00GP3oool00`00
0000oooo0?ooo`070?ooo`@0000000L0FZ7f06:UmP1JXOH0HZGf05ZQmP1RYOH0FZ7f0080000000D0
cLKN0<G6e@3=a]h0aLKE0<g6gP020000000Q0;C:iP2mb^H0]<[V0;g:iP2db^H0_L[V0;C:iP2mb^H0
]<[V0;g:iP2db^H0_L[V0;C:iP000000aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0
cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0080000002H0_LKV0;g6gP2ma^H0
_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0
_LKV0;g6gP2ma^H000000;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0
[;KN0;BjgP2/]]h0];[N0:bfgP<00000503oool400000400oooo001O0?ooo`030000003oool0oooo
00L0oooo00<0000006:Zo`1RYOH01`0000001@1JXOH0HZGf0000003=b]h0aLKE008000000240_L[^
0;C:iP2mb^h0]<[V0;g:kP2db^H0_L[^0;C:iP2mb^h0]<[V0;g:kP2db^H0_L[^0;C:iP2mb^h0]<[V
00000035_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D0aKk=0<fne@35_/d0cKkE
0<Fnc@3=_]D00P000000902dd^h0]=;f0000002mb^H0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV
0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0000002/]]h0];[V
0:bfgP2d^^H0[;KN0;BjiP2/]]h0];[V0:bfgP2d^^H0[;KN0;BjiP@000004`3oool4000004@0oooo
001P0?ooo`800000203oool02@000000HZ[f06:UmP1RZ_H0HZGf06:ZmP1RYOH0HZ[f06:UmP050000
000Q0;C:iP2mb^H0]<[V0;g:iP2db^H0_L[V0;C:iP2mb^H0]<[V0;g:iP2db^H0_L[V0;C:iP2mb^H0
]<[V0;g:iP2db^H0_L[V0;C:iP000000aKk=0<fnc@35_/d0cKk=0<Fnc@3=_/d0aKk=0<fnc@35_/d0
cKk=0<Fnc@3=_/d0aKk=008000000280]=;^0;C>kP2dd^h0]<k^0;CBkP000000_LKV0;g6gP2ma^H0
_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0
_LKV0;g6gP000000[;KN0;BjgP2/]]h0];[N0:bfgP2d^]h0[;KN1000000C0?ooo`@00000B03oool0
0680oooo00<000000?ooo`3oool01P3oool030000000HZ[o06:UmP1RZ_l0HZGf06:Zo`1RYOH0HZ[o
06:UmP1RZ_l0HZGf06:Zo`80000001h0]<[V0;g:kP2db^H0_L[^0;C:iP2mb^h0]<[V0;g:kP2db^H0
_L[^0;C:iP2mb^h0]<[V0;g:kP2db^H0_L[^0;C:iP2mb^h0]<[V00000035_/d0cKkE0<Fnc@3=_]D0
aKk=0<fne@35_/d0cKkE0<Fnc@3=_]D20000000P0;CBkP2dd_H0]=;^0;CBmP2dd^h0]=;f0;CBkP2d
d_H000000;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2m
a^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2ma^H000000:bfgP2d^^H4000001<0oooo1000001<0?ooo`00
H`3oool00`000000oooo0?ooo`060?ooo`0Y0000001RZ_H0HZGf06:ZmP1RYOH0HZ[f06:UmP1RZ_H0
HZGf06:ZmP1RYOH0HZ[f06:UmP000000]<[V0;g:iP2db^H0_L[V0;C:iP2mb^H0]<[V0;g:iP2db^H0
_L[V0;C:iP2mb^H0]<[V0;g:iP2db^H0_L[V0;C:iP2mb^H000000<fnc@35_/d0cKk=0<Fnc@3=_/d0
aKk=0<fnc@35_/d00P0000007`2dd^h0]<k^0;CBkP2dc^h0]=;^0;C>kP2dd^h0]<k^0;CBkP2dc^h0
]=;^0000002ma^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H0
_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H00`00000C0?ooo`@00000D03oool006@0oooo00<000000?oo
o`3oool01P3oool20000000T06:UmP1RZ_l0HZGf06:Zo`1RYOH0HZ[o06:UmP1RZ_l0HZGf06:Zo`1R
YOH000000;C:iP2mb^h0]<[V0;g:kP2db^H0_L[^0;C:iP2mb^h0]<[V0;g:kP2db^H0_L[^0;C:iP2m
b^h0]<[V0;g:kP2db^H0_L[^0000003=_]D0aKk=0<fne@35_/d0cKkE0P0000007`2dd^h0]=;f0;CB
kP2dd_H0]=;^0;CBmP2dd^h0]=;f0;CBkP2dd_H0]=;^0;CBmP2dd^h0]=;f0000002mb^H0_LKV0;g:
iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2ma^H00P00000C
0?ooo`<00000E03oool006D0oooo00<000000?ooo`3oool01`3oool030000000HZGf06:ZmP1RYOH0
HZ[f06:UmP1RZ_H0HZGf06:ZmP1RYOH0HZ[f06:UmP80000001@0_L[V0;C:iP2mb^H0]<[V0;g:iP2d
b^H0_L[V0;C:iP2mb^H0]<[V0;g:iP2db^H0_L[V0;C:iP2mb^H0]<[V0;g:iP000000cKk=0<Fnc@80
000001h0]=;^0;C>kP2dd^h0]<k^0;CBkP2dc^h0]=;^0;C>kP2dd^h0]<k^0;CBkP2dc^h0]=;^0;C>
kP2dd^h0]<k^0;CBkP000000_LKV0;g6gP2ma^H0_LKN0;g6iP2ma]h0_LKV0;g6gP2ma^H0_LKN0;g6
iP2ma]h300000140oooo1000001G0?ooo`00IP3oool2000000P0oooo01d0000006:UmP1RZ_l0HZGf
06:Zo`1RYOH0HZ[o06:UmP1RZ_l0HZGf06:Zo`1RYOH0HZ[o0000002mb^h0]<[V0;g:kP2db^H0_L[^
0;C:iP2mb^h0]<[V0;g:kP2db^H0_L[^0;C:iP2mb^h0]<[V0;g:kP040000000N0;CBkP2dd_H0]=;^
0;CBmP2dd^h0]=;f0;CBkP2dd_H0]=;^0;CBmP2dd^h0]=;f0;CBkP2dd_H0]=;^0;CBmP2dd^h0]=;f
0;CBkP2dd_H000000;g:iP2ma^H0_L[V0;g6iP2mb^H0_LKV0;g:iP2ma^H0_L[V0P00000@0?ooo`@0
0000F`3oool006P0oooo00<000000?ooo`3oool01P3oool060000000HZGf06:ZmP1RYOH0HZ[f06:U
mP1RZ_H0HZGf06:ZmP1RYOH0HZ[f06:UmP1RZ_H000000;g:iP2db^H0_L[V0;C:iP2mb^H0]<[V0;g:
iP2db^H0_L[V0;C:iP@0000000<0W=;o09c>mP2Ld_l00P0000006P2dc^h0]=;^0;C>kP2dd^h0]<k^
0;CBkP2dc^h0]=;^0;C>kP2dd^h0]<k^0;CBkP2dc^h0]=;^0;C>kP2dd^h0]<k^0;CBkP2dc^h0]=;^
0000002ma^H0_LKN0;g6iP2ma]h0_LKV0`00000>0?ooo`@00000G`3oool006T0oooo00<000000?oo
o`3oool01P3oool03@000000HZGf06:Zo`1RYOH0HZ[o06:UmP1RZ_l0HZGf06:Zo`1RYOH0HZ[o06:U
mP1RZ_l00P0000000`2db^H0_L[^0;C:iP050000000Q09c>mP2Td_l0W<kf0:CBo`2Lc_H0Y=;o09c>
mP2Td_l0W<kf0000002dd^h0]=;f0;CBkP2dd_H0]=;^0;CBmP2dd^h0]=;f0;CBkP2dd_H0]=;^0;CB
mP2dd^h0]=;f0;CBkP2dd_H0]=;^0;CBmP2dd^h0]=;f0000002mb^H0_LKV008000003@3oool40000
06<0oooo001Z0?ooo`030000003oool0oooo00H0oooo4@0000008`2Lc_H0W=;o09c>mP2Ld_l0W<kf
09cBo`2Lc_H0W=;o09c>mP2Ld_l0W<kf09cBo`2Lc_H0W=;o09c>mP000000]<k^0;CBkP2dc^h0]=;^
0;C>kP2dd^h0]<k^0;CBkP2dc^h0]=;^0;C>kP2dd^h0]<k^0;CBkP2dc^h0]=;^0;C>kP2dd^h0]<k^
00<000002`3oool4000006L0oooo001[0?ooo`8000001`3oool03P000000J[;f07>bo`1Z/_H0Lk;o
06ZbmP1c/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f0P000000402Td_l0W<kf0:CBo`2Lc_H0
Y=;o09c>mP2Td_l0W<kf0:CBo`2Lc_H0Y=;o09c>mP2Td_l0W<kf0:CBo`2Lc_H20000000=0;CBmP2d
d^h0]=;f0;CBkP2dd_H0]=;^0;CBmP2dd^h0]=;f0;CBkP2dd_H0]=;^0;CBmP04000000X0oooo1000
001[0?ooo`00K@3oool00`000000oooo0?ooo`050?ooo`80000002P0J[;o06ZbmP1Z/_l0J[;f06Zb
o`1Z/_H0J[;o06ZbmP1Z/_l0J[;f06Zbo`1Z/_H0J[;o0000002Ld_l0W<kf09cBo`2Lc_H0W=;o09c>
mP2Ld_l0W<kf09cBo`2Lc_H0W=;o09c>mP2Ld_l0W<kf09cBo`2Lc_H0W=;o0000002dd^h0]<k^0;CB
kP2dc^h0]=;^0;C>kP2dd^h0]<k^1000000;0?ooo`<00000K`3oool006h0oooo00<000000?ooo`3o
ool01P3oool080000000Lk;o06ZbmP1c/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f07>bo`1Z
/_H0Lk;o0000002Td_l0W<kf0:CBo`2Lc_H0Y=;o09c>mP2Td_l0W<kf0:CBo`2Lc_H0Y=;o09c>mP2T
d_l0W<kf0:CBo`2Lc_H0Y=;o0P0000000`2dd^h0]=;f00000003000000/0oooo1000001b0?ooo`00
K`3oool00`000000oooo0?ooo`060?ooo`0>0000001Z/_l0J[;f06Zbo`1Z/_H0J[;o06ZbmP1Z/_l0
J[;f06Zbo`1Z/_H0J[;o06ZbmP1Z/_l20000000?09c>mP2Ld_l0W<kf09cBo`2Lc_H0W=;o09c>mP2L
d_l0W<kf09cBo`2Lc_H0W=;o09c>mP2Ld_l0W<kf00@000002`3oool4000007H0oooo001`0?ooo`80
00001`3oool06P000000Lk;o06ZbmP1c/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l0J[;f07>bo`1Z
/_H0Lk;o06ZbmP000000W<kf0:CBo`2Lc_H0Y=;o09c>mP2Td_l0W<kf0:CBo`2Lc_H0Y=;o1000000;
0?ooo`@00000NP3oool00780oooo00<000000?ooo`3oool01@3oool20000000=06ZbmP1Z/_l0J[;f
06Zbo`1Z/_H0J[;o06ZbmP1Z/_l0J[;f06Zbo`1Z/_H0J[;o06ZbmP020000000309cBo`2Lc_H0W=;o
00D000002`3oool4000007h0oooo001c0?ooo`030000003oool0oooo00H0oooo00T0000006ZbmP1c
/_l0J[;f07>bo`1Z/_H0Lk;o06ZbmP1c/_l02@00000<0?ooo`@00000PP3oool007@0oooo00<00000
0?ooo`3oool01P3oool800000180oooo0`0000260?ooo`00M@3oool2000000H0oooo00<000000?oo
o`3oool04`3oool4000008T0oooo001g0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3o
ool03`3oool4000008d0oooo001h0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
2`3oool400000940oooo001i0?ooo`050000003oool0oooo0?ooo`0000002@3oool4000009D0oooo
001j0?ooo`80000000<0oooo0000003oool0103oool4000009T0oooo001l0?ooo`80000000<0oooo
0000000000000P00002M0?ooo`00O@3oool200000:40oooo003o0?ooob40oooo003o0?ooob40oooo
003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {232.562, 0}} -> {-0.0879705, -0.00360347, \
0.00372121, 0.00372121}}]
}, Open  ]],

Cell[BoxData[
    FormBox[
      StyleBox[
        FrameBox[
          StyleBox[GridBox[{
                {"\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"option name\\\\\\\"\\\", \
\\\"SO10\\\"]\\) \"\>", "\<\"\\!\\(\\*StyleBox[\\\"\\\\\\\"default \
value\\\\\\\"\\\", \\\"SO10\\\"]\\) \"\>", "\[Null]"},
                {"\<\"\\!\\(\\* \
StyleBox[\\\"\\\\\\\"DifferenceOrder\\\\\\\"\\\",\\n\\\"MR\\\"]\\) \"\>", "\<\
\"\\!\\(\\* StyleBox[\\\"\\\\\\\"4\\\\\\\"\\\",\\n\\\"MR\\\"]\\) \"\>", \
\(\(asymptotic\)\(\ \)\(order\)\(\ \)\(of\)\(\ \)\(the\)\(\ \)\(error\)\(\ \)\
\)},
                {"\<\"\\!\\(\\* \
StyleBox[\\\"\\\\\\\"PeriodicInterpolation\\\\\\\"\\\",\\n\\\"MR\\\"]\\) \
\"\>", "\<\"\\!\\(\\* \
StyleBox[\\\"\\\\\\\"False\\\\\\\"\\\",\\n\\\"MR\\\"]\\) \"\>", 
                  RowBox[{
                  "whether", " ", "to", " ", "consider", " ", "the", " ", 
                    "values", " ", "as", " ", "those", " ", "of", " ", 
                    StyleBox["a",
                      FontSlant->"Plain"], " ", "periodic", " ", "function", 
                    " ", "with", " ", "the", " ", "period", " ", "equal", " ",
                     "to", " ", "the", " ", "interval", " ", "enclosed", " ", 
                    "by", " ", "the", " ", "grid"}]}
                },
              ColumnAlignments->{Left}],
            GridBoxOptions->{RowLines->{True, False}}]],
        "3ColumnBox"], TraditionalForm]], "Text",
  FormatType->StandardForm,
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["Options for NDSolve`FiniteDifferenceDerivative.", "Caption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["\<\
This approximates the derivatives for the sine function on the \
random grid defined above, assuming that the function repeats periodically.\
\
\>", "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`FiniteDifferenceDerivative[1, rgrid, \ Sin[rgrid], \ 
      PeriodicInterpolation \[Rule] True]\)], "Input",
  CellLabel->"In[28]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \({0.9763950803634552`, 
      0.2559211638833165`, \(-0.8783201219513552`\), \
\(-0.9321130826184576`\), \(-0.975268635427687`\), \(-0.9829759598563446`\), \
\(-0.5186667134192805`\), \(-0.16942903234823478`\), 0.2869269871509373`, 
      0.5056178660405766`, 0.5874607222783306`, 
      0.9763950803634552`}\)], "Output",
  CellLabel->"Out[28]="]
}, Open  ]],

Cell["\<\
When using PeriodicInterpolation\[Rule]True, you can omit the last \
point in the values since it should always be the same as the first. This \
feature is useful when solving a PDE with periodic boundary conditions.\
\>", \
"Text",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell["\<\
This generates second order finite difference formulas for the \
first derivative of a symbolic function.\
\>", "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`FiniteDifferenceDerivative[
      1, \ {x\_\(-1\), x\_0, x\_1}, \ {f\_\(-1\), f\_0, \ f\_1}, \ 
      DifferenceOrder \[Rule] 2]\)], "Input",
  CellLabel->"In[29]:=",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[BoxData[
    \({\(f\_1\ \((x\_\(-1\) - x\_0)\)\)\/\(\((\(-x\_\(-1\)\) + x\_1)\)\ \
\((\(-x\_0\) + x\_1)\)\) + \(f\_0\ \((\(-x\_\(-1\)\) + x\_1)\)\)\/\(\((\(-x\_\
\(-1\)\) + x\_0)\)\ \((\(-x\_0\) + x\_1)\)\) + \(f\_\(-1\)\ \((\(-1\) - \
\(\(-x\_\(-1\)\) + x\_1\)\/\(\(-x\_\(-1\)\) + x\_0\))\)\)\/\(\(-x\_\(-1\)\) + \
x\_1\), \(f\_1\ \((\(-x\_\(-1\)\) + x\_0)\)\)\/\(\((\(-x\_\(-1\)\) + x\_1)\)\ \
\((\(-x\_0\) + x\_1)\)\) - \(f\_\(-1\)\ \((\(-x\_0\) + x\_1)\)\)\/\(\((\(-x\_\
\(-1\)\) + x\_0)\)\ \((\(-x\_\(-1\)\) + x\_1)\)\) + \(f\_0\ \((\(-1\) + \
\(\(-x\_0\) + x\_1\)\/\(\(-x\_\(-1\)\) + x\_0\))\)\)\/\(\(-x\_0\) + x\_1\), \
\(-\(\(f\_\(-1\)\ \((x\_0 - x\_1)\)\)\/\(\((\(-x\_\(-1\)\) + 
                    x\_0)\)\ \((\(-x\_\(-1\)\) + 
                    x\_1)\)\)\)\) - \(f\_0\ \((\(-x\_\(-1\)\) + x\_1)\)\)\/\(\
\((\(-x\_\(-1\)\) + x\_0)\)\ \((\(-x\_0\) + x\_1)\)\) + \(f\_1\ \((\(-x\_\(-1\
\)\) + x\_0)\)\ \((\(\(-x\_\(-1\)\) + x\_1\)\/\(\(-x\_\(-1\)\) + x\_0\) + \
\(\(-x\_0\) + x\_1\)\/\(\(-x\_\(-1\)\) + x\_0\))\)\)\/\(\((\(-x\_\(-1\)\) + x\
\_1)\)\ \((\(-x\_0\) + x\_1)\)\)}\)], "Output",
  CellLabel->"Out[29]="]
}, Open  ]],

Cell["\<\
Fourth order differences typically provide a good balance between \
truncation (approximation) error and roundoff error for machine precision. \
However, there are some applications where fourth order differences produce \
excessive oscillation (Gibb's phenomena), so second order differences are \
better. Also, for high precision, higher order differences may be \
appropriate. Even values of DifferenceOrder use centered formulas, which \
typically have smaller error coefficients than non-centered formulas, so even \
values are recommended when appropriate.\
\>", "Text",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["NDSolve`FiniteDifferenceDerivativeFunction", "Subsubsection",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1", "PDE:1.2.3"}],

Cell[TextData[{
  "When computing the solution to a PDE, it is common to repeatedly apply the \
same finite difference approximation to different values on the same grid. A \
significant savings can be made by storing the necessary weight computations \
and applying them to the changing data.",
  " ",
  "When you omit the (third) argument with function values in ",
  StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
  ", the result will be an ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
  ", which is a data object that stores the weight computations in a \
efficient form for future repeated use."
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {
            StyleBox[
              RowBox[{"NDSolve`FiniteDifferenceDerivative", "[", 
                
                StyleBox[\({m1, \ m2, \  ... }, \ {grid1, \ grid2, \  ... }\),
                  
                  FontSlant->"Italic"], "]"}],
              "MR"], Cell[TextData[{
              
              "compute the finite difference weights needed to approximate \
the partial derivative of order (",
              StyleBox["m1, m2, ...", "MR",
                FontSlant->"Italic"],
              
              ") for the function of n variables on the tensor product grid \
defined by the outer product of (",
              StyleBox["grid1, grid2, ...", "MR",
                FontSlant->"Italic"],
              ").  The result is returned as an ",
              StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
              " object."
            }], "Text"]},
          {
            RowBox[{
              StyleBox["NDSolve`FiniteDifferenceDerivativeFunction",
                "MR"], 
              StyleBox["[",
                "MR"], 
              RowBox[{
                StyleBox[
                  RowBox[{"Derivative", "[", 
                    StyleBox["m",
                      FontSlant->"Italic"], "]"}],
                  "MR"], 
                StyleBox[",",
                  "MR"], 
                StyleBox["data",
                  "TI"]}], 
              StyleBox["]",
                "MR"]}], Cell[TextData[{
              
              "a data object that contains the weights and other data needed \
to quickly approximate the ",
              Cell[BoxData[
                  \(TraditionalForm\`m\^\(\(th\)\(\ \)\)\)]],
              
              "order derivative of a function.  In the standard output form, \
only the ",
              StyleBox["Derivative[", "MR"],
              StyleBox["m", "MR",
                FontSlant->"Italic"],
              StyleBox["]", "MR"],
              " operator it approximates is shown. "
            }], "Text"]},
          {
            RowBox[{
              RowBox[{
                StyleBox["NDSolve`FiniteDifferenceDerivativeFunction",
                  "MR"], 
                StyleBox["[",
                  "MR"], 
                StyleBox["data",
                  "TI"], 
                StyleBox["]",
                  "MR"]}], 
              StyleBox["[",
                "MR"], 
              StyleBox["values",
                "MR",
                FontSlant->"Italic"], 
              StyleBox["]",
                "MR"]}], Cell[TextData[{
              "approximate the derivative of the function that takes on ",
              StyleBox["values", "MR",
                FontSlant->"Italic"],
              " on the grid used to determine ",
              StyleBox["data", "MR",
                FontSlant->"Italic"],
              "."
            }], "Text"]}
          }], TextForm]], "DefinitionBox"],

Cell["Computing finite difference weights for repeated use. ", "Caption"],

Cell["\<\
This defines a uniform grid with 25 points on the unit interval and \
evaluates the sine function with one period on the grid.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(n\  = \ 24;\)\ \), "\[IndentingNewLine]", 
    \(\(grid\  = \ N[Range[0, n]/n];\)\), "\[IndentingNewLine]", 
    \(values\  = \ Sin[2\ \[Pi]\ grid]\)}], "Input",
  CellLabel->"In[30]:="],

Cell[BoxData[
    \({0.`, 0.25881904510252074`, 0.49999999999999994`, 0.7071067811865475`, 
      0.8660254037844386`, 0.9659258262890683`, 1.`, 0.9659258262890683`, 
      0.8660254037844387`, 0.7071067811865476`, 0.49999999999999994`, 
      0.258819045102521`, 
      1.2246063538223773`*^-16, \(-0.25881904510252035`\), \
\(-0.5000000000000001`\), \(-0.7071067811865475`\), \(-0.8660254037844384`\), \
\(-0.9659258262890683`\), \(-1.`\), \(-0.9659258262890684`\), \
\(-0.8660254037844386`\), \(-0.7071067811865477`\), \(-0.5000000000000004`\), \
\(-0.2588190451025207`\), \(-2.4492127076447545`*^-16\)}\)], "Output",
  CellLabel->"Out[32]="]
}, Open  ]],

Cell[TextData[{
  "This defines an ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
  ", which can be repeatedly applied to different values on the grid to \
approximate the second derivative."
}], "MathCaption",
  CellTags->{"b:7.1.1", "ndsg:2.0.1.1"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(fddf\  = \ 
      NDSolve`FiniteDifferenceDerivative[Derivative[2], \ grid]\)], "Input",
  CellLabel->"In[33]:="],

Cell[BoxData[
    TagBox[\(NDSolve`FiniteDifferenceDerivativeFunction[Derivative[2], 
        "<>"]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[33]="]
}, Open  ]],

Cell["\<\
Note that the standard output form is abbreviated and only shows \
the derivative operators that are approximated. \
\>", "Text"],

Cell["\<\
This computes the approximation to the second derivative of the \
sine function.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(fddf[values]\)], "Input",
  CellLabel->"In[34]:="],

Cell[BoxData[
    \({0.07202673938179843`, \(-10.224832144498144`\), \
\(-19.73818479164852`\), \(-27.91400862897582`\), \(-34.187538908318665`\), \
\(-38.13124490863909`\), \(-39.47636958329719`\), \(-38.13124490863901`\), \
\(-34.18753890831874`\), \(-27.914008628975914`\), \(-19.73818479164831`\), \
\(-10.217236279663524`\), 3.3968661217187446`*^-13, 10.21723627966258`, 
      19.738184791649115`, 27.914008628975846`, 34.18753890831835`, 
      38.13124490863927`, 39.476369583297085`, 38.13124490863926`, 
      34.18753890831844`, 27.914008628975765`, 19.738184791649203`, 
      10.224832144497753`, \(-0.07202673938650651`\)}\)], "Output",
  CellLabel->"Out[34]="]
}, Open  ]],

Cell[TextData[{
  "This function is only applicable for values defined on the particular grid \
used to construct it. If your problem requires changing the grid, you will \
need to use ",
  StyleBox["NDSolve`FiniteDifferenceDerivative", "MR"],
  " to generate weights each time the grid changes.",
  " ",
  "However, when you can use ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
  " objects, evaluation will be substantially faster. "
}], "Text"],

Cell["\<\
This compares timings for computing the Laplacian with the function \
just defined and with the definition of the previous section. A loop is used \
to repeat the calculation in each case because it is too fast for the \
differences to show up with Timing. \
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(repeats\  = \ 1000;\)\), "\[IndentingNewLine]", 
    \({First[Timing[Do[fddf[values], \ {repeats}]]], \[IndentingNewLine]First[
        Timing[Do[
            NDSolve`FiniteDifferenceDerivative[Derivative[2], \ grid, \ 
              values], {repeats}]]]}\)}], "Input",
  CellLabel->"In[35]:="],

Cell[BoxData[
    \({0.020000000000000018`\ Second, 
      0.7110000000000001`\ Second}\)], "Output",
  CellLabel->"Out[36]="]
}, Open  ]],

Cell[TextData[{
  "An ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
  " can be used repeatedly in many situations.",
  " ",
  "As a simple example, consider a collocation method for solving the \
boundary value problem"
}], "Text"],

Cell[BoxData[
    FormBox[
      RowBox[{
        RowBox[{
          RowBox[{
            SubscriptBox["u", 
              StyleBox["xx",
                FontSlant->"Italic"]], " ", "+", " ", \(\(sin(x)\)\ u\)}], 
          " ", "=", " ", \(\[Lambda]\ u\)}], ";", 
        " ", \(u(0)\  = \ \(u(1)\  = \ 0\)\)}], 
      TraditionalForm]], "DisplayMath"],

Cell["\<\
on the unit interval. (This simple method is not necessarily the \
best way to solve this particular problem, but it is useful as an example.)\
\
\>", "Text"],

Cell["\<\
This defines a function that will have all components zero at an \
approximate solution of the boundary value problem. Using the intermediate \
vector v and setting its endpoints (parts {1,-1}) to 0 is a fast and simple \
trick to enforce the boundary conditions. Evaluation is prevented except for \
numbers \[Lambda] because this would not work otherwise. (Also, because of \
the Listability of Times, Sin[2 Pi grid] u would thread componentwise.)\
\>", \
"MathCaption"],

Cell[BoxData[{
    \(\(Clear[fun];\)\), "\[IndentingNewLine]", 
    \(fun[u_, \ \[Lambda]_?NumberQ]\  := 
      Module[{n\  = \ Length[u], \ 
          v\  = \ \ fddf[
                u]\  + \((\ 
                  Sin[grid]\  - \ \[Lambda])\)\ u}, \
\[IndentingNewLine]v[\([{1, \(-1\)}]\)]\  = \ 0. ; \[IndentingNewLine]{v, \ 
          u . u\  - \ 1}]\)}], "Input",
  CellLabel->"In[37]:="],

Cell["\<\
This uses FindRoot to find an approximate eigenfunction using the \
constant coefficient case for a starting value and shows a plot of the \
eigenfunction.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(s4\  = \ 
        FindRoot[
          fun[u, \ \[Lambda]], \ {u, 
            values}, \ {\[Lambda], \ \(-4\)\ \[Pi]\^2}];\)\), "\
\[IndentingNewLine]", 
    \(\(ListPlot[Transpose[{grid, \ u\  /. \ s4}], \ 
        PlotLabel \[Rule] ToString[Last[s4]]];\)\)}], "Input",
  CellLabel->"In[39]:="],

Cell[BoxData[
    \(FindRoot::"symdv" \(\(:\)\(\ \)\) 
      "Unable to automatically compute the symbolic derivative for a function \
of vector arguments. Numerical approximations to derivatives will be used \
instead."\)], "Message",
  CellLabel->"From In[39]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.308611 1.0195 [
[.21429 .29611 -9 -9 ]
[.21429 .29611 9 0 ]
[.40476 .29611 -9 -9 ]
[.40476 .29611 9 0 ]
[.59524 .29611 -9 -9 ]
[.59524 .29611 9 0 ]
[.78571 .29611 -9 -9 ]
[.78571 .29611 9 0 ]
[.97619 .29611 -3 -9 ]
[.97619 .29611 3 0 ]
[.01131 .00276 -24 -4.5 ]
[.01131 .00276 0 4.5 ]
[.01131 .10471 -24 -4.5 ]
[.01131 .10471 0 4.5 ]
[.01131 .20666 -24 -4.5 ]
[.01131 .20666 0 4.5 ]
[.01131 .41056 -18 -4.5 ]
[.01131 .41056 0 4.5 ]
[.01131 .51251 -18 -4.5 ]
[.01131 .51251 0 4.5 ]
[.01131 .61446 -18 -4.5 ]
[.01131 .61446 0 4.5 ]
[.5 .63053 -41 0 ]
[.5 .63053 41 12.5625 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .30861 m
.21429 .31486 L
s
[(0.2)] .21429 .29611 0 1 Mshowa
.40476 .30861 m
.40476 .31486 L
s
[(0.4)] .40476 .29611 0 1 Mshowa
.59524 .30861 m
.59524 .31486 L
s
[(0.6)] .59524 .29611 0 1 Mshowa
.78571 .30861 m
.78571 .31486 L
s
[(0.8)] .78571 .29611 0 1 Mshowa
.97619 .30861 m
.97619 .31486 L
s
[(1)] .97619 .29611 0 1 Mshowa
.125 Mabswid
.07143 .30861 m
.07143 .31236 L
s
.11905 .30861 m
.11905 .31236 L
s
.16667 .30861 m
.16667 .31236 L
s
.2619 .30861 m
.2619 .31236 L
s
.30952 .30861 m
.30952 .31236 L
s
.35714 .30861 m
.35714 .31236 L
s
.45238 .30861 m
.45238 .31236 L
s
.5 .30861 m
.5 .31236 L
s
.54762 .30861 m
.54762 .31236 L
s
.64286 .30861 m
.64286 .31236 L
s
.69048 .30861 m
.69048 .31236 L
s
.7381 .30861 m
.7381 .31236 L
s
.83333 .30861 m
.83333 .31236 L
s
.88095 .30861 m
.88095 .31236 L
s
.92857 .30861 m
.92857 .31236 L
s
.25 Mabswid
0 .30861 m
1 .30861 L
s
.02381 .00276 m
.03006 .00276 L
s
[(-0.3)] .01131 .00276 1 0 Mshowa
.02381 .10471 m
.03006 .10471 L
s
[(-0.2)] .01131 .10471 1 0 Mshowa
.02381 .20666 m
.03006 .20666 L
s
[(-0.1)] .01131 .20666 1 0 Mshowa
.02381 .41056 m
.03006 .41056 L
s
[(0.1)] .01131 .41056 1 0 Mshowa
.02381 .51251 m
.03006 .51251 L
s
[(0.2)] .01131 .51251 1 0 Mshowa
.02381 .61446 m
.03006 .61446 L
s
[(0.3)] .01131 .61446 1 0 Mshowa
.125 Mabswid
.02381 .02315 m
.02756 .02315 L
s
.02381 .04354 m
.02756 .04354 L
s
.02381 .06393 m
.02756 .06393 L
s
.02381 .08432 m
.02756 .08432 L
s
.02381 .1251 m
.02756 .1251 L
s
.02381 .14549 m
.02756 .14549 L
s
.02381 .16588 m
.02756 .16588 L
s
.02381 .18627 m
.02756 .18627 L
s
.02381 .22705 m
.02756 .22705 L
s
.02381 .24744 m
.02756 .24744 L
s
.02381 .26783 m
.02756 .26783 L
s
.02381 .28822 m
.02756 .28822 L
s
.02381 .329 m
.02756 .329 L
s
.02381 .34939 m
.02756 .34939 L
s
.02381 .36978 m
.02756 .36978 L
s
.02381 .39017 m
.02756 .39017 L
s
.02381 .43095 m
.02756 .43095 L
s
.02381 .45134 m
.02756 .45134 L
s
.02381 .47173 m
.02756 .47173 L
s
.02381 .49212 m
.02756 .49212 L
s
.02381 .5329 m
.02756 .5329 L
s
.02381 .55329 m
.02756 .55329 L
s
.02381 .57368 m
.02756 .57368 L
s
.02381 .59407 m
.02756 .59407 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
gsave
.5 .63053 -102 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.5625 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding WindowsANSIEncoding def
        currentdict
    end

    newfontname exch definefont pop
} def

63.000 12.813 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.813 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(l) show
75.000 12.813 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(->) show
93.000 12.813 moveto
(-) show
99.000 12.813 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(39.0144) show
141.000 12.813 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.008 w
.02381 .30861 Mdot
.06349 .38457 Mdot
.10317 .45541 Mdot
.14286 .51633 Mdot
.18254 .56323 Mdot
.22222 .5929 Mdot
.2619 .60332 Mdot
.30159 .59376 Mdot
.34127 .56486 Mdot
.38095 .51856 Mdot
.42063 .45798 Mdot
.46032 .38724 Mdot
.5 .31114 Mdot
.53968 .23487 Mdot
.57937 .16364 Mdot
.61905 .10231 Mdot
.65873 .05508 Mdot
.69841 .02519 Mdot
.7381 .01472 Mdot
.77778 .02438 Mdot
.81746 .05353 Mdot
.85714 .10019 Mdot
.89683 .16116 Mdot
.93651 .23226 Mdot
.97619 .30861 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[39]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`003@3oool2000000@0oooo0P0000040?ooo`800000o`3oool60?ooo`00
303oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000000@0oooo00<000000?oo
o`3oool0o@3oool000`0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`030?oo
o`<00000o@3oool000H0oooo100000020?ooo`040000003oool0oooo000000/0oooo00<000000?oo
o`3oool00P3oool00`000000oooo0?ooo`3m0?ooo`00303oool010000000oooo0?ooo`0000080?oo
o`040000003oool0oooo000000@0oooo00<000000?ooo`3oool0o@3oool000d0oooo0P00000:0?oo
o`8000001@3oool00`000000oooo0?ooo`2c0?ooo`030000003oool0oooo04L0oooo000P0?ooo`03
0000003oool0oooo0;<0oooo00<000000?ooo`3oool0A`3oool00200oooo0P00002Z0?ooo`030000
003oool0oooo0140oooo00<000000?ooo`3oool0?@3oool00200oooo00<000000?ooo`3oool0Z@3o
ool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo03d0oooo000P0?ooo`030000003oool0
oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0
oooo000P0?ooo`800000oP3oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo00<00000
0?ooo`3oool0a`3oool00`000000oooo0?ooo`0c0?ooo`00803oool00`000000oooo0?ooo`2N0?oo
o`030000003oool0oooo02H0oooo00<000000?ooo`3oool0<`3oool00200oooo00<000000?ooo`3o
ool0WP3oool00`000000oooo0?ooo`1L0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803o
ool200000?h0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo
0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo
000P0?ooo`800000oP3oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo00<000000?oo
o`3oool0o@3oool000d0oooo0P0000040?ooo`8000000`3oool4000000@0oooo00<000000?ooo`3o
ool0d@3oool00`000000oooo0?ooo`0Y0?ooo`00303oool010000000oooo0?ooo`0000080?ooo`03
0000003oool0oooo00D0oooo00<000000?ooo`3oool0U03oool00`000000oooo0?ooo`0j0?ooo`03
0000003oool0oooo02T0oooo000<0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3o
ool0103oool3000009@0oooo00<000000?ooo`3oool0IP3oool000H0oooo100000020?ooo`040000
003oool0oooo000000X0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`3m0?ooo`00
303oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000000@0oooo00<000000?oo
o`3oool0o@3oool000d0oooo0P00000:0?ooo`8000001@3oool00`000000oooo0?ooo`3m0?ooo`00
803oool00`000000oooo0?ooo`3m0?ooo`00803oool200000?h0oooo000P0?ooo`030000003oool0
oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0
oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`800000oP3oool00200oooo00<00000
0?ooo`3oool0o@3oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo00<000000?ooo`3o
ool0o@3oool00200oooo00<000000?ooo`3oool0f`3oool00`000000oooo0?ooo`0O0?ooo`00803o
ool00`000000oooo0?ooo`2:0?ooo`030000003oool0oooo04h0oooo00<000000?ooo`3oool07`3o
ool00200oooo0P00002;0?ooo`030000003oool0oooo0700oooo000P0?ooo`030000003oool0oooo
0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo
000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`800000oP3oool00200oooo00<000000?oo
o`3oool0o@3oool00200oooo00<000000?ooo`3oool0o@3oool000d0oooo0P0000040?ooo`800000
0`3oool5000000<0oooo00<000000?ooo`3oool0o@3oool000`0oooo00@000000?ooo`3oool00000
2P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo0?d0oooo000<0?ooo`040000003o
ool0oooo000000X0oooo00<000000?ooo`3oool00`3oool300000?d0oooo00060?ooo`@000000P3o
ool010000000oooo0?ooo`00000:0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
o@3oool000`0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo0?d0oooo000=0?ooo`8000002P3oool2000000D0oooo00<000000?ooo`3oool0o@3o
ool00200oooo00<000000?ooo`3oool0o@3oool00200oooo0P00003n0?ooo`00803oool00`000000
oooo0?ooo`3U0?ooo`030000003oool0oooo01D0oooo000P0?ooo`030000003oool0oooo0800oooo
00<000000?ooo`3oool0HP3oool00`000000oooo0?ooo`0E0?ooo`00803oool00`000000oooo0?oo
o`200?ooo`030000003oool0oooo07X0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?oo
o`800000oP3oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo00<000000?ooo`3oool0
o@3oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo00<000000?ooo`3oool0o@3oool0
0200oooo00<000000?ooo`3oool09`3oool2000000@0oooo0P0000030?ooo`@000008@3oool20000
00@0oooo0P0000050?ooo`<000008@3oool2000000@0oooo0P0000030?ooo`<000008P3oool20000
00@0oooo0P0000040?ooo`800000:03oool5000000X0oooo000P0?ooo`8000009`3oool010000000
oooo0?ooo`0000080?ooo`030000003oool0oooo0240oooo00@000000?ooo`3oool000002`3oool0
0`000000oooo0?ooo`0O0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool00000
803oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000002T0oooo00<000000?oo
o`3oool02P3oool00200oooo00<000000?ooo`3oool09P3oool010000000oooo0?ooo`0000090?oo
o`030000003oool0oooo0200oooo00@000000?ooo`3oool00000203oool500000200oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`00000P0?ooo`040000003oool0oooo000000P0
oooo00@000000?ooo`3oool00000:@3oool00`000000oooo0?ooo`0:0?ooo`00803oool00`000000
oooo0?ooo`0V0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool07`3oool01000
0000oooo0?ooo`0000080?ooo`040000003oool0oooo00000240oooo00@000000?ooo`3oool00000
203oool300000240oooo00@000000?ooo`3oool000002@3oool2000002X0oooo00<000000?ooo`3o
ool02P3oool00200oooo00<000000?ooo`3oool09P3oool010000000oooo0?ooo`0000080?ooo`04
0000003oool0oooo00000200oooo00@000000?ooo`3oool000002@3oool00`000000oooo0000000Q
0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool0803oool010000000oooo0?oo
o`0000080?ooo`040000003oool0oooo000002T0oooo00<000000?ooo`3oool02P3oool00200oooo
00<000000?ooo`3oool09`3oool2000000X0oooo0P00000R0?ooo`8000002`3oool200000280oooo
0P00000:0?ooo`<000008@3oool2000000X0oooo0P00000Y0?ooo`800000303oool00200oooo0P00
003n0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3m0?oo
o`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3_0?ooo`030000
003oool0oooo00/0oooo000J0?ooool000000@0000060?ooo`00803oool00`000000oooo0?ooo`09
0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`09
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`09
0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`09
0?ooo`8000002`3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00T0oooo00<00000
0?ooo`3oool02@3oool00`000000oooo0?ooo`0:0?ooo`00803oool00`000000oooo0?ooo`0^0?oo
o`030000003oool0oooo02d0oooo00<000000?ooo`3oool0;P3oool00`000000oooo0?ooo`0]0?oo
o`030000003oool0oooo02h0oooo00<000000?ooo`3oool02P3oool00200oooo00<000000?ooo`3o
ool0o@3oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo0P00003n0?ooo`00803oool0
0`000000oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool00`000000
oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool200000?h0oooo000P
0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`03
0000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003o
ool0oooo0?d0oooo000P0?ooo`800000oP3oool00200oooo00<000000?ooo`3oool0o@3oool00200
oooo00<000000?ooo`3oool0o@3oool00200oooo00<000000?ooo`3oool01`3oool00`000000oooo
0?ooo`1R0?ooo`030000003oool0oooo08h0oooo000P0?ooo`030000003oool0oooo00L0oooo00<0
00000?ooo`3oool0HP3oool00`000000oooo0?ooo`2>0?ooo`00803oool200000?h0oooo000P0?oo
o`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000=0?ooo`800000
103oool2000000<0oooo1@0000030?ooo`030000003oool0oooo0?d0oooo000<0?ooo`040000003o
ool0oooo000000X0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`3m0?ooo`00303o
ool010000000oooo0?ooo`00000:0?ooo`030000003oool0oooo00<0oooo0`00003m0?ooo`00303o
ool010000000oooo0?ooo`00000:0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
o@3oool000`0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`030?ooo`030000
003oool0oooo0?d0oooo000=0?ooo`8000002P3oool2000000D0oooo00<000000?ooo`3oool0o@3o
ool00200oooo00<000000?ooo`3oool0o@3oool00200oooo0P00003n0?ooo`00803oool00`000000
oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?oo
o`3m0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool200000?h0oooo000P0?ooo`03
0000003oool0oooo0140oooo00<000000?ooo`3oool0CP3oool00`000000oooo0?ooo`2H0?ooo`00
803oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo04h0oooo00<000000?ooo`3oool0
V03oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo00<000000?ooo`3oool0o@3oool0
0200oooo0P00003n0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool00`000000oooo
0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3m
0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool200000?h0oooo000P0?ooo`030000
003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000=0?ooo`800000103oool2
000000<0oooo100000040?ooo`030000003oool0oooo0?d0oooo000<0?ooo`040000003oool0oooo
000000P0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`3m0?ooo`00303oool01000
0000oooo0?ooo`0000090?ooo`030000003oool0oooo00@0oooo0`00000K0?ooo`030000003oool0
oooo0=l0oooo000<0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0K0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool0XP3oool0
00`0oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000040?ooo`030000003o
ool0oooo05P0oooo00<000000?ooo`3oool0XP3oool000d0oooo0P00000:0?ooo`8000001@3oool0
0`000000oooo0?ooo`3m0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool200000?h0
oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P
0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`80
0000oP3oool00200oooo00<000000?ooo`3oool0o@3oool00200oooo00<000000?ooo`3oool09@3o
ool00`000000oooo0?ooo`3E0?ooo`00803oool00`000000oooo0?ooo`0U0?ooo`030000003oool0
oooo02H0oooo00<000000?ooo`3oool0[03oool00200oooo00<000000?ooo`3oool0CP3oool00`00
0000oooo0?ooo`2/0?ooo`00803oool200000?h0oooo000P0?ooo`030000003oool0oooo0?d0oooo
000P0?ooo`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo0?d0oooo000P0?oo
o`030000003oool0oooo0?d0oooo000P0?ooo`030000003oool0oooo02l0oooo00<000000?ooo`3o
ool04P3oool00`000000oooo0?ooo`2f0?ooo`00803oool200000300oooo00<000000?ooo`3oool0
4P3oool00`000000oooo0?ooo`2f0?ooo`00803oool00`000000oooo0?ooo`3m0?ooo`00803oool0
0`000000oooo0?ooo`0i0?ooo`030000003oool0oooo0<40oooo000=0?ooo`800000103oool20000
00@0oooo0P0000050?ooo`030000003oool0oooo03T0oooo00<000000?ooo`3oool0`@3oool000`0
oooo00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000040?ooo`030000003oool0
oooo0?d0oooo000<0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00`3oool3
00000?d0oooo000<0?ooo`040000003oool0oooo000000/0oooo00<000000?ooo`3oool00P3oool0
0`000000oooo0?ooo`3m0?ooo`00303oool010000000oooo0?ooo`0000080?ooo`040000003oool0
oooo00000?l0oooo1@3oool000d0oooo0P00000:0?ooo`800000o`3oool60?ooo`00o`3ooolQ0?oo
o`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00LP3oool30000
0080oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0@0?ooo`8000000`3oool30000
00@0oooo0P0000040?ooo`8000000`3oool5000000<0oooo0`0000030?ooo`<00000G`3oool007@0
oooo00<000000?ooo`0000002@3oool4000000<0oooo00<000000?ooo`3oool0203oool400000080
oooo00@000000?ooo`3oool00000103oool00`000000oooo0?ooo`070?ooo`040000003oool0oooo
000000@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`030000003oool0
oooo05h0oooo001e0?ooo`8000004@3oool200000100oooo00<000000?ooo`3oool00P3oool30000
00P0oooo00@000000?ooo`3oool00000103oool010000000oooo0?ooo`3oool5000000030?ooo`00
0000000000<00000G`3oool007H0oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`0A
0?ooo`040000003oool0oooo00000080oooo00<000000?ooo`3oool01P3oool010000000oooo0?oo
o`0000040?ooo`050000003oool0oooo0?ooo`0000000P3oool010000000oooo0?ooo`0000020?oo
o`030000003oool0oooo05h0oooo001f0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3o
ool03`3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000P0oooo00@00000
0?ooo`3oool00000103oool00`000000oooo0?ooo`020?ooo`030000003oool0000000<0oooo00<0
00000?ooo`000000H03oool007@0oooo0P00000S0?ooo`800000103oool3000000T0oooo0P000004
0?ooo`8000001@3oool2000000@0oooo0P00001P0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00
\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.135189, -0.317336, \
0.004123, 0.00385156}}]
}, Open  ]],

Cell[TextData[{
  "The ",
  StyleBox["FindRoot::symdv", "MR"],
  " message does not mean that FindRoot is unable to properly solve the \
problem. However, giving consideration to its import can lead to substantial \
improvements in the algorithm.",
  " ",
  "These will be explored in the section below on differentiation matrices."
}], "Text"],

Cell[TextData[{
  "The setup for this problem is so simple, it is easy to compare various \
alternatives.",
  " ",
  "For example, to compare the solution above, which used the default fourth \
order differences, to the usual use of second order differences, all that \
needs to be changed is the ",
  StyleBox["DifferenceOrder", "MR"],
  "."
}], "Text"],

Cell["\<\
This solves the boundary value problem using second order \
differences and shows a plot of the difference between it and the fourth \
order solution.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(fddf\  = \ 
        NDSolve`FiniteDifferenceDerivative[Derivative[2], \ grid, \ 
          DifferenceOrder \[Rule] 2];\)\), "\[IndentingNewLine]", 
    \(\(s2\  = \ 
        FindRoot[
          fun[u, \ \[Lambda]], \ {u, 
            values}, \ {\[Lambda], \ \(-4\)\ \[Pi]\^2}];\)\), "\
\[IndentingNewLine]", 
    \(\(ListPlot[
        Transpose[{grid, \ \((u\  /. \ s4)\)\  - \ \((u\  /. \ 
                  s2)\)}]];\)\)}], "Input",
  CellLabel->"In[41]:="],

Cell[BoxData[
    \(FindRoot::"symdv" \(\(:\)\(\ \)\) 
      "Unable to automatically compute the symbolic derivative for a function \
of vector arguments. Numerical approximations to derivatives will be used \
instead."\)], "Message",
  CellLabel->"From In[41]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.952381 0.402232 13577.1 [
[.21429 .38973 -9 -9 ]
[.21429 .38973 9 0 ]
[.40476 .38973 -9 -9 ]
[.40476 .38973 9 0 ]
[.59524 .38973 -9 -9 ]
[.59524 .38973 9 0 ]
[.78571 .38973 -9 -9 ]
[.78571 .38973 9 0 ]
[.97619 .38973 -3 -9 ]
[.97619 .38973 3 0 ]
[.01131 .13069 -48 -4.5 ]
[.01131 .13069 0 4.5 ]
[.01131 .26646 -48 -4.5 ]
[.01131 .26646 0 4.5 ]
[.01131 .538 -42 -4.5 ]
[.01131 .538 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.21429 .40223 m
.21429 .40848 L
s
[(0.2)] .21429 .38973 0 1 Mshowa
.40476 .40223 m
.40476 .40848 L
s
[(0.4)] .40476 .38973 0 1 Mshowa
.59524 .40223 m
.59524 .40848 L
s
[(0.6)] .59524 .38973 0 1 Mshowa
.78571 .40223 m
.78571 .40848 L
s
[(0.8)] .78571 .38973 0 1 Mshowa
.97619 .40223 m
.97619 .40848 L
s
[(1)] .97619 .38973 0 1 Mshowa
.125 Mabswid
.07143 .40223 m
.07143 .40598 L
s
.11905 .40223 m
.11905 .40598 L
s
.16667 .40223 m
.16667 .40598 L
s
.2619 .40223 m
.2619 .40598 L
s
.30952 .40223 m
.30952 .40598 L
s
.35714 .40223 m
.35714 .40598 L
s
.45238 .40223 m
.45238 .40598 L
s
.5 .40223 m
.5 .40598 L
s
.54762 .40223 m
.54762 .40598 L
s
.64286 .40223 m
.64286 .40598 L
s
.69048 .40223 m
.69048 .40598 L
s
.7381 .40223 m
.7381 .40598 L
s
.83333 .40223 m
.83333 .40598 L
s
.88095 .40223 m
.88095 .40598 L
s
.92857 .40223 m
.92857 .40598 L
s
.25 Mabswid
0 .40223 m
1 .40223 L
s
.02381 .13069 m
.03006 .13069 L
s
[(-0.00002)] .01131 .13069 1 0 Mshowa
.02381 .26646 m
.03006 .26646 L
s
[(-0.00001)] .01131 .26646 1 0 Mshowa
.02381 .538 m
.03006 .538 L
s
[(0.00001)] .01131 .538 1 0 Mshowa
.125 Mabswid
.02381 .02207 m
.02756 .02207 L
s
.02381 .04923 m
.02756 .04923 L
s
.02381 .07638 m
.02756 .07638 L
s
.02381 .10354 m
.02756 .10354 L
s
.02381 .15784 m
.02756 .15784 L
s
.02381 .185 m
.02756 .185 L
s
.02381 .21215 m
.02756 .21215 L
s
.02381 .23931 m
.02756 .23931 L
s
.02381 .29362 m
.02756 .29362 L
s
.02381 .32077 m
.02756 .32077 L
s
.02381 .34792 m
.02756 .34792 L
s
.02381 .37508 m
.02756 .37508 L
s
.02381 .42939 m
.02756 .42939 L
s
.02381 .45654 m
.02756 .45654 L
s
.02381 .4837 m
.02756 .4837 L
s
.02381 .51085 m
.02756 .51085 L
s
.02381 .56516 m
.02756 .56516 L
s
.02381 .59231 m
.02756 .59231 L
s
.25 Mabswid
.02381 0 m
.02381 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.008 w
.02381 .40223 Mdot
.06349 .44589 Mdot
.10317 .52174 Mdot
.14286 .56736 Mdot
.18254 .57355 Mdot
.22222 .54028 Mdot
.2619 .47308 Mdot
.30159 .38184 Mdot
.34127 .27933 Mdot
.38095 .17943 Mdot
.42063 .09535 Mdot
.46032 .03801 Mdot
.5 .01472 Mdot
.53968 .02832 Mdot
.57937 .07696 Mdot
.61905 .15439 Mdot
.65873 .2508 Mdot
.69841 .35412 Mdot
.7381 .45156 Mdot
.77778 .53124 Mdot
.81746 .58375 Mdot
.85714 .60332 Mdot
.89683 .58838 Mdot
.93651 .53822 Mdot
.97619 .40223 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[41]:=",
  ImageSize->{366.938, 226.812},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40005^0000hR000`400?l00000o`00003oo`3ooom_
0?ooo`00o`3ooom_0?ooo`00o`3ooom_0?ooo`00o`3ooom_0?ooo`00o`3ooom_0?ooo`00o`3ooom_
0?ooo`00o`3ooom_0?ooo`00o`3ooom_0?ooo`00o`3ooom_0?ooo`00o`3ooom_0?ooo`00o`3ooom_
0?ooo`00o`3ooom_0?ooo`00o`3ooom_0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo
000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh
0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0
oooo>03oool003@0oooo00<000000?ooo`3oool0U@3oool00`000000oooo0?ooo`2O0?ooo`00=03o
ool00`000000oooo0?ooo`2E0?ooo`030000003oool0oooo09l0oooo000d0?ooo`800000o`3oooli
0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0:80
oooo00<000000?ooo`3oool0TP3oool003@0oooo00<000000?ooo`3oool0XP3oool00`000000oooo
0?ooo`2B0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0
oooo08P0oooo00<000000?ooo`3oool0[03oool003@0oooo00<000000?ooo`3oool0R03oool00`00
0000oooo0?ooo`2/0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000
003oool0oooo0?l0oooo>03oool003@0oooo0P00003o0?ooocT0oooo000d0?ooo`030000003oool0
oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000
oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<0
00000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?oo
o`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00
=03oool200000:l0oooo00<000000?ooo`3oool0QP3oool003@0oooo00<000000?ooo`3oool0[P3o
ool00`000000oooo0?ooo`260?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?oo
o`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00
=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo07/0oooo00<0
00000?ooo`3oool0^@3oool003@0oooo00<000000?ooo`3oool0N`3oool00`000000oooo0?ooo`2i
0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`800000o`3oooli0?ooo`00
=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03o
ool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?oo
ocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool000P0oooo0P0000040?ooo`800000
103oool2000000@0oooo0P0000040?ooo`800000103oool2000000<0oooo100000050?ooo`030000
003oool0oooo0?l0oooo>03oool000L0oooo00@000000?ooo`3oool00000203oool010000000oooo
0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool0
10000000oooo0?ooo`0000020?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0o`3o
oolh0?ooo`001`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo
00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo
000000<0oooo00<000000?ooo`3oool01@3oool300000?l0oooo>03oool000030?ooo`0000000000
008000000P3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@0
00000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
00@0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`3o0?ooocP0oooo00070?ooo`04
0000003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000
oooo0?ooo`0000050?ooo`030000003oool0oooo0?l0oooo>03oool000P0oooo0P00000:0?ooo`80
0000103oool2000000@0oooo0P0000040?ooo`800000103oool2000000H0oooo00<000000?ooo`3o
ool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003o
ool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`00
0000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0;/0oooo00<000000?ooo`3o
ool0N@3oool003@0oooo0P00002l0?ooo`030000003oool0oooo07T0oooo000d0?ooo`030000003o
ool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`00
0000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo
00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d
0?ooo`030000003oool0oooo06l0oooo00<000000?ooo`3oool0a@3oool003@0oooo00<000000?oo
o`3oool0K`3oool00`000000oooo0?ooo`350?ooo`00=03oool200000?l0oooo>@3oool003@0oooo
00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d
0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?oo
o`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo
>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool200000?l0oooo>@3oool0
03@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0
oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3o
oolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo
0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo
0?ooo`3o0?ooocP0oooo000d0?ooo`800000o`3oooli0?ooo`00=03oool00`000000oooo0?ooo`3o
0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3o
ool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`380?ooo`030000003oool0oooo06`0oooo
000d0?ooo`030000003oool0oooo0<P0oooo00<000000?ooo`3oool0K03oool003@0oooo00<00000
0?ooo`3oool0o`3ooolh0?ooo`00203oool2000000@0oooo0P0000040?ooo`800000103oool20000
00@0oooo0P0000040?ooo`8000000`3oool5000000@0oooo00<000000?ooo`3oool0o`3ooolh0?oo
o`001`3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo00000080oooo00@00000
0?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000@0
oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`3o0?ooocP0oooo00070?ooo`040000
003oool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`000002
0?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool00`000000oooo
0?ooo`040?ooo`<00000o`3ooolh0?ooo`0000<0oooo0000000000000P0000020?ooo`040000003o
ool0oooo000000P0oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000020?oo
o`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000103oool00`000000oooo0?oo
o`040?ooo`030000003oool0oooo0?l0oooo>03oool000L0oooo00@000000?ooo`3oool00000203o
ool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3o
ool000000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo00@0oooo00<00000
0?ooo`3oool0o`3ooolh0?ooo`00203oool2000000X0oooo0P0000040?ooo`800000103oool20000
00@0oooo0P0000040?ooo`8000001P3oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`03
0000003oool0oooo0680oooo00<000000?ooo`3oool0dP3oool003@0oooo00<000000?ooo`3oool0
HP3oool00`000000oooo0?ooo`3B0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d
0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?oo
o`00=03oool200000?l0oooo>@3oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03o
ool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool0
03@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0
oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3o
oolh0?ooo`00=03oool200000?l0oooo>@3oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?oo
o`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo
>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o
0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3o
ool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`800000o`3o
ooli0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo
0=D0oooo00<000000?ooo`3oool0G`3oool003@0oooo00<000000?ooo`3oool0e@3oool00`000000
oooo0?ooo`1O0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003o
ool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0<`3oool2000000@0oooo0P000003
0?ooo`@00000;P3oool2000000@0oooo0P0000050?ooo`<00000;@3oool2000000@0oooo0P000003
0?ooo`<00000<03oool2000000@0oooo0P0000040?ooo`800000=03oool5000000D0oooo000d0?oo
o`030000003oool0oooo0380oooo00@000000?ooo`3oool00000203oool00`000000oooo0?ooo`0^
0?ooo`040000003oool0oooo000000/0oooo00<000000?ooo`3oool0:`3oool010000000oooo0?oo
o`0000080?ooo`040000003oool0oooo000002h0oooo00@000000?ooo`3oool00000203oool01000
0000oooo0?ooo`00000e0?ooo`030000003oool0oooo00D0oooo000d0?ooo`030000003oool0oooo
0380oooo00@000000?ooo`3oool000002@3oool00`000000oooo0?ooo`0]0?ooo`040000003oool0
oooo000000P0oooo1@00000/0?ooo`040000003oool0oooo000000P0oooo00@000000?ooo`3oool0
0000;P3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000003D0oooo00<00000
0?ooo`3oool01@3oool003@0oooo0P00000c0?ooo`040000003oool0oooo000000X0oooo00<00000
0?ooo`3oool0;03oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000002d0oooo
00@000000?ooo`3oool00000203oool3000002l0oooo00@000000?ooo`3oool000002@3oool20000
03H0oooo00<000000?ooo`3oool01@3oool003@0oooo00<000000?ooo`3oool0<P3oool010000000
oooo0?ooo`0000080?ooo`040000003oool0oooo000002d0oooo00@000000?ooo`3oool000002@3o
ool00`000000oooo0000000]0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool0
;P3oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000003D0oooo00<000000?oo
o`3oool01@3oool003@0oooo00<000000?ooo`3oool0<`3oool2000000X0oooo0P00000D0?ooo`03
0000003oool0oooo01P0oooo0P00000;0?ooo`800000;P3oool2000000X0oooo0`00000_0?ooo`80
00002P3oool2000003D0oooo0P0000070?ooo`00=03oool00`000000oooo0?ooo`1E0?ooo`030000
003oool0oooo0=l0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<00000
0?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`03
0000003oool0oooo0?l0oooo>03oool003<0oooo0P00003o0?oooc40oooo00<000000?ooo`3oool0
1P3oool002`0ooooo`00001300000000=03oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0=0?ooo`030000003oool0
oooo00`0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo00d0oooo00<000000?ooo`3oool01@3oool003@0oooo00<000000?ooo`3oool0>P3oool00`00
0000oooo0?ooo`0j0?ooo`030000003oool0oooo03X0oooo00<000000?ooo`3oool0>`3oool00`00
0000oooo0?ooo`0j0?ooo`030000003oool0oooo00D0oooo000d0?ooo`030000003oool0oooo0?l0
oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?oo
o`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?oo
o`3oool0o`3ooolh0?ooo`00=03oool200000?l0oooo>@3oool003@0oooo00<000000?ooo`3oool0
o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0
oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000
oooo0?ooo`090?ooo`030000003oool0oooo0?l0oooo;03oool003@0oooo00<000000?ooo`3oool0
2@3oool00`000000oooo0?ooo`3o0?ooob`0oooo000d0?ooo`030000003oool0oooo0>40oooo00<0
00000?ooo`3oool0D`3oool003@0oooo00<000000?ooo`3oool0h@3oool00`000000oooo0?ooo`1C
0?ooo`00=03oool200000?l0oooo>@3oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00
=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03o
ool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`180?oo
o`030000003oool0oooo0>`0oooo000d0?ooo`030000003oool0oooo04P0oooo00<000000?ooo`3o
ool0k03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?oo
o`3o0?ooocP0oooo000d0?ooo`800000o`3oooli0?ooo`00=03oool00`000000oooo0?ooo`3o0?oo
ocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0
o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0
oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000
oooo0?ooo`3o0?ooocP0oooo000d0?ooo`800000o`3oooli0?ooo`00=03oool00`000000oooo0?oo
o`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?oo
o`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo0?l0
oooo803oool003@0oooo00<000000?ooo`3oool05@3oool00`000000oooo0?ooo`3o0?ooob00oooo
000d0?ooo`030000003oool0oooo0?l0oooo>03oool000P0oooo0P0000040?ooo`800000103oool2
000000@0oooo0P0000040?ooo`800000103oool2000000<0oooo1@0000040?ooo`030000003oool0
oooo0>h0oooo00<000000?ooo`3oool0AP3oool000L0oooo00@000000?ooo`3oool00000203oool0
10000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool0
00000P3oool010000000oooo0?ooo`0000040?ooo`030000003oool0oooo00@0oooo00<000000?oo
o`3oool0kP3oool00`000000oooo0?ooo`160?ooo`001`3oool010000000oooo0?ooo`0000080?oo
o`040000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?oo
o`0000020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool0103oool300000?l0
oooo8P3oool00`000000oooo0?ooo`0C0?ooo`001`3oool010000000oooo0?ooo`0000080?ooo`04
0000003oool0oooo00000080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`00
00020?ooo`040000003oool0oooo000000@0oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`0l0?ooo`030000003oool0oooo0>80oooo00<000000?ooo`3oool04`3oool000L0oooo00@0
00000?ooo`3oool00000203oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo0000
0080oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000040?ooo`030000003o
ool0oooo00@0oooo00<000000?ooo`3oool0?03oool00`000000oooo0?ooo`3h0?ooo`00203oool2
000000X0oooo0P0000040?ooo`800000103oool2000000@0oooo0P0000040?ooo`8000001P3oool0
0`000000oooo0?ooo`3o0?ooocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0
oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo
000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh
0?ooo`00=03oool2000002<0oooo00<000000?ooo`3oool0o`3ooolC0?ooo`00=03oool00`000000
oooo0?ooo`0R0?ooo`030000003oool0oooo0?l0oooo4`3oool003@0oooo00<000000?ooo`3oool0
;`3oool00`000000oooo0?ooo`3o0?ooo`H0oooo000d0?ooo`030000003oool0oooo02l0oooo00<0
00000?ooo`3oool0o`3oool60?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?oo
o`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0n`3oool00`000000
oooo0?ooo`0i0?ooo`00=03oool00`000000oooo0?ooo`3k0?ooo`030000003oool0oooo01L0oooo
00<000000?ooo`3oool07`3oool003@0oooo00<000000?ooo`3oool0o`3ooolF0?ooo`030000003o
ool0oooo01l0oooo000d0?ooo`800000o`3oooli0?ooo`00=03oool00`000000oooo0?ooo`3o0?oo
ocP0oooo000d0?ooo`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0
o`3oool90?ooo`030000003oool0oooo02`0oooo000d0?ooo`030000003oool0oooo0?l0oooo2@3o
ool00`000000oooo0?ooo`0/0?ooo`00=03oool00`000000oooo0?ooo`3o0?ooocP0oooo000d0?oo
o`030000003oool0oooo0?l0oooo>03oool003@0oooo00<000000?ooo`3oool0o`3ooolh0?ooo`00
=03oool00`000000oooo0?ooo`3o0?ooocP0oooo003o0?ooofl0oooo003o0?ooofl0oooo003o0?oo
ofl0oooo003o0?ooofl0oooo003o0?ooofl0oooo003o0?ooofl0oooo003o0?ooofl0oooo003o0?oo
ofl0oooo003o0?ooofl0oooo003o0?ooofl0oooo003o0?ooofl0oooo003o0?ooofl0oooo003o0?oo
ofl0oooo0000\
\>"],
  ImageRangeCache->{{{0, 365.938}, {225.812, 0}} -> {-0.16986, -3.27261, \
0.00326522, 2.29042}}]
}, Open  ]],

Cell["\<\
One way to determine which is the better solution is to study the \
convergence as the grid is refined. This will be considered to some extent in \
the section on differentiation matrices below. \
\>", "Text"],

Cell[TextData[{
  "While the most vital information about an ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
  " object, the derivative order, is displayed in its output form, sometimes \
it is useful to extract this and other information from an ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
  ", say for use in a program. The structure of the way the data is stored \
may change between versions of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ", so extracting the information by using parts of the expression is not \
recommended.",
  " ",
  "A better alternative is to use any of the several method functions \
provided for this purpose. "
}], "Text"],

Cell[TextData[{
  "Let ",
  StyleBox["FDDF", "MR",
    FontSlant->"Italic"],
  " represent an ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction[", "MR"],
  StyleBox["data", "MR",
    FontSlant->"Italic"],
  StyleBox["]", "MR"],
  " object."
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[{
              StyleBox["FDDF", "MR",
                FontSlant->"Italic"],
              StyleBox["@\"DerivativeOrder\"[]", "MR"]
            }]], Cell[TextData[{
              "get the derivative order which ",
              StyleBox["FDDF", "MR",
                FontSlant->"Italic"],
              " approximates."
            }], "Text"]},
          {Cell[TextData[{
              StyleBox["FDDF", "MR",
                FontSlant->"Italic"],
              StyleBox["@\"DifferenceOrder\"[]", "MR"]
            }]], 
            Cell["\<\
get the list with the difference order used for the \
approximation in each dimension.  \
\>", "Text"]},
          {Cell[TextData[{
              StyleBox["FDDF", "MR",
                FontSlant->"Italic"],
              StyleBox["@\"PeriodicInterpolation\"[]", "MR"]
            }]], Cell[TextData[{
              "get the list with elements ",
              StyleBox["True", "MR"],
              " or ",
              StyleBox["False", "MR"],
              
              " indicating whether periodic interpolation is used for each \
dimension."
            }]]},
          {Cell[TextData[{
              StyleBox["FDDF", "MR",
                FontSlant->"Italic"],
              StyleBox["@\"Coordinates\"[]", "MR"]
            }]], \(get\ the\ list\ with\ the\ grid\ coordinates\ in\ each\ \(\
\(dimension\)\(.\)\)\)},
          {Cell[TextData[{
              StyleBox["FDDF", "MR",
                FontSlant->"Italic"],
              StyleBox["@\"Grid\"[]", "MR"]
            }]], \(form\ the\ tensor\ of\ the\ grid\ points . \ \ This\ is\ \
the\ outer\ product\ of\ the\ grid\ \(\(coordinates\)\(.\)\(\ \)\)\)},
          {Cell[TextData[{
              StyleBox["FDDF", "MR",
                FontSlant->"Italic"],
              StyleBox["@\"DifferentiationMatrix\"[]", "MR"]
            }]], 
            RowBox[{
            "compute", " ", "the", " ", "sparse", " ", "differentiation", " ",
               "matrix", " ", 
              StyleBox["mat",
                "MR",
                FontSlant->"Italic"], " ", "such", " ", "that", " ", 
              RowBox[{
                StyleBox[
                  RowBox[{"m", 
                    StyleBox["at",
                      "MR"]}],
                  FontSlant->"Italic",
                  FontVariations->{"StrikeThrough"->True}], 
                StyleBox[".",
                  "MR"], 
                StyleBox[
                  RowBox[{"Flatten", "[", 
                    StyleBox["values",
                      FontSlant->"Italic"], "]"}],
                  "MR"]}], " ", "is", " ", "equivalent", " ", "to", " ", 
              RowBox[{
                StyleBox["Flatten",
                  "MR"], 
                StyleBox["[",
                  "MR"], 
                RowBox[{
                  StyleBox["FDDF",
                    "MR",
                    FontSlant->"Italic"], "[", 
                  StyleBox["values",
                    "MR",
                    FontSlant->"Italic"], "]"}], "]"}]}]}
          }], TextForm]], "DefinitionBox",
  GridBoxOptions->{ColumnWidths->{0.4, 0.6}}],

Cell[TextData[{
  "Method functions for exacting information from an ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction[", "MR"],
  StyleBox["data", "MR",
    FontSlant->"Italic"],
  StyleBox["]", "MR"],
  " object."
}], "Caption"],

Cell[TextData[{
  "Any of the method functions that return a list with an element for each of \
the dimensions can be used with an integer argument ",
  StyleBox["dim",
    FontSlant->"Italic"],
  ", which will return only the value for that particular dimension such that \
",
  StyleBox["FDDF", "MR",
    FontSlant->"Italic"],
  StyleBox["@", "MR"],
  StyleBox["method", "MR",
    FontSlant->"Italic"],
  StyleBox["[dim]", "MR"],
  " = ",
  StyleBox["(FDDF@", "MR"],
  StyleBox["method", "MR",
    FontSlant->"Italic"],
  StyleBox["[])[[", "MR"],
  StyleBox["dim", "MR",
    FontSlant->"Italic"],
  StyleBox["]]", "MR"],
  "."
}], "Text"],

Cell["\<\
The examples below show how you might use some of these \
methods.\
\>", "Text"],

Cell["\<\
Here is an NDSolve`FiniteDifferenceDerivativeFunction object \
created with random grids having between 10 and 16 points in each \
dimension.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(fddf\  = \ 
      NDSolve`FiniteDifferenceDerivative[Derivative[0, 1, 2], \ 
        Table[Sort[
            Join[{0. , 1. }, \ 
              Table[Random[], {Random[Integer, {8, 14}]}]]], {3}], \ 
        PeriodicInterpolation \[Rule] True]\)], "Input",
  CellLabel->"In[44]:="],

Cell[BoxData[
    TagBox[\(NDSolve`FiniteDifferenceDerivativeFunction[Derivative[0, 1, 2], 
        "<>"]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[44]="]
}, Open  ]],

Cell["This shows the dimensions of the outer product grid. ", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Dimensions[tpg\  = \ fddf@Grid[]]\)], "Input",
  CellLabel->"In[45]:="],

Cell[BoxData[
    \({11, 16, 13, 3}\)], "Output",
  CellLabel->"Out[45]="]
}, Open  ]],

Cell[TextData[{
  "Note that the rank of the grid point tensor is one more than the \
dimensionality of the tensor product. This is because the 3 coordinates \
defining each point are in a list themselves.",
  " ",
  "If you have a function that depends on the grid variables, you can use ",
  StyleBox["Apply[f, fddf@Grid[], {n}]", "MR"],
  " where n = ",
  StyleBox["Length[fddf@DerivativeOrder[]]", "MR"],
  " is the dimensionality of the space in which you are approximating the \
derivative."
}], "Text"],

Cell[TextData[{
  "This defines a Gaussian function of 3 variables and applies it to the grid \
on which the ",
  StyleBox["NDSolve`FiniteDifferenceDerivativeFunction", "MR"],
  " is defined."
}], "MathCaption"],

Cell[BoxData[{
    \(\(f\  = \ 
        Function[{x, y, z}, \ 
          Exp[\(-\((\((x -  .5)\)^2\  + \ \((y -  .5)\)^2\  + \ \((z -  \
.5)\)^2)\)\)]];\)\), "\[IndentingNewLine]", 
    \(\(values\  = \ 
        Apply[f, \ 
          fddf@Grid[], \ {Length[fddf@DerivativeOrder[]]}];\)\)}], "Input",
  CellLabel->"In[46]:="],

Cell["\<\
This shows a 3-dimensional plot of the grid points colored \
according to the scaled value of the derivative.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Show[
        Module[{dvals\  = \ fddf[values], maxval, \ 
            minval}, \[IndentingNewLine]maxval\  = \ 
            Max[dvals]; \[IndentingNewLine]minval\  = \ 
            Min[dvals]; \[IndentingNewLine]Graphics3D[
            MapThread[{Hue[\((#2\  - \ minval)\)/\((maxval\  - \ 
                          minval)\)], \ Point[#1]} &, \ {fddf@Grid[], 
                fddf[values]}, \ 
              Length[fddf@DerivativeOrder[]]]]]];\)\)], "Input",
  CellLabel->"In[48]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: 1.0855 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics3D
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.0567234 1.16608 1.26852e-017 1.16608 [
[ 0 0 0 0 ]
[ 1 1.0855 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.08846 .24555 m
0 .80374 L
s
0 .80374 m
.40037 1.0855 L
s
.40037 1.0855 m
.41799 .58158 L
s
.41799 .58158 m
.08846 .24555 L
s
.67245 0 m
.92713 .41497 L
s
.92713 .41497 m
1 .94814 L
s
1 .94814 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.08846 .24555 m
0 .80374 L
s
0 .80374 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.67245 0 m
.08846 .24555 L
s
.41799 .58158 m
.92713 .41497 L
s
.92713 .41497 m
1 .94814 L
s
1 .94814 m
.40037 1.0855 L
s
.40037 1.0855 m
.41799 .58158 L
s
0 0 m
1 0 L
1 1.0855 L
0 1.0855 L
closepath
clip
newpath
.269 0 1 r
.01 w
.42216 .58164 Mdot
.29 0 1 r
.42208 .58389 Mdot
.323 0 1 r
.43974 .57589 Mdot
.118 1 0 r
.42173 .59426 Mdot
.344 0 1 r
.43968 .57814 Mdot
.345 0 1 r
.44744 .57337 Mdot
.368 0 1 r
.44738 .57563 Mdot
.156 1 0 r
.43939 .58853 Mdot
.172 1 0 r
.44711 .58603 Mdot
1 .605 0 r
.39947 .55845 Mdot
1 .571 0 r
.39938 .56071 Mdot
1 .871 0 r
.39425 .55311 Mdot
1 .518 0 r
.41722 .55254 Mdot
1 .841 0 r
.39415 .55538 Mdot
.46 0 1 r
.39895 .57115 Mdot
1 .483 0 r
.41714 .5548 Mdot
1 .481 0 r
.42499 .54995 Mdot
1 .445 0 r
.42491 .55222 Mdot
1 .794 0 r
.41204 .54716 Mdot
.273 0 1 r
.39371 .56583 Mdot
1 .763 0 r
.41195 .54943 Mdot
.522 0 1 r
.41677 .56526 Mdot
1 .761 0 r
.41982 .54456 Mdot
.557 0 1 r
.53593 .54444 Mdot
1 .73 0 r
.41975 .54683 Mdot
.583 0 1 r
.53594 .54671 Mdot
.547 0 1 r
.42457 .56268 Mdot
.98 1 0 r
.38315 .54176 Mdot
1 .992 0 r
.38304 .54403 Mdot
.328 0 1 r
.41156 .5599 Mdot
.35 0 1 r
.41938 .55731 Mdot
.322 1 0 r
.53597 .55719 Mdot
1 .949 0 r
.40101 .53573 Mdot
.168 0 1 r
.38255 .55452 Mdot
1 .92 0 r
.40092 .53801 Mdot
1 .919 0 r
.40884 .53309 Mdot
1 .889 0 r
.40875 .53537 Mdot
.807 1 0 r
.37143 .52978 Mdot
.218 0 1 r
.40049 .54851 Mdot
.832 1 0 r
.37131 .53206 Mdot
.24 0 1 r
.40834 .54588 Mdot
.871 1 0 r
.38938 .52366 Mdot
.047 0 1 r
.37078 .54257 Mdot
.898 1 0 r
.38927 .52595 Mdot
.899 1 0 r
.39723 .52099 Mdot
1 .139 0 r
.51435 .52019 Mdot
.925 1 0 r
.39714 .52327 Mdot
1 .098 0 r
.51434 .52247 Mdot
0 1 .859 r
.41808 .70291 Mdot
.092 0 1 r
.3888 .53648 Mdot
1 .459 0 r
.50939 .51461 Mdot
.111 0 1 r
.39669 .53381 Mdot
1 .423 0 r
.50937 .5169 Mdot
.789 0 1 r
.5143 .53302 Mdot
.498 1 0 r
.35242 .51035 Mdot
0 1 .858 r
.43634 .69737 Mdot
.518 1 0 r
.35229 .51264 Mdot
0 1 .858 r
.44433 .69495 Mdot
.563 0 1 r
.50931 .52746 Mdot
.55 1 0 r
.3705 .50409 Mdot
0 .171 1 r
.35168 .52321 Mdot
.571 1 0 r
.37039 .50639 Mdot
1 .639 0 r
.49882 .50274 Mdot
0 1 .858 r
.4172 .72933 Mdot
1 .605 0 r
.4988 .50503 Mdot
.572 1 0 r
.37842 .50136 Mdot
.594 1 0 r
.37831 .50365 Mdot
0 .134 1 r
.36984 .51697 Mdot
0 1 .858 r
.4356 .72384 Mdot
.436 0 1 r
.4987 .51562 Mdot
0 .119 1 r
.37779 .51425 Mdot
.211 1 0 r
.3362 .49376 Mdot
0 1 .857 r
.44366 .72144 Mdot
.228 1 0 r
.33605 .49606 Mdot
0 1 .882 r
.39444 .68055 Mdot
1 .848 0 r
.48766 .49019 Mdot
1 .817 0 r
.48763 .4925 Mdot
.253 1 0 r
.3544 .48739 Mdot
0 .373 1 r
.33538 .50667 Mdot
.27 1 0 r
.35426 .48969 Mdot
0 1 .881 r
.38899 .67541 Mdot
0 1 .883 r
.41287 .67485 Mdot
.27 1 0 r
.36237 .4846 Mdot
.288 1 0 r
.36224 .4869 Mdot
.29 0 1 r
.48749 .50312 Mdot
0 1 .857 r
.41597 .76593 Mdot
0 1 .883 r
.42094 .67235 Mdot
0 .343 1 r
.35366 .50032 Mdot
0 1 .881 r
.40747 .66967 Mdot
.617 0 1 r
.7377 .47847 Mdot
.643 0 1 r
.73786 .48077 Mdot
0 .331 1 r
.36166 .49754 Mdot
0 1 .884 r
.39334 .70717 Mdot
0 1 .881 r
.41556 .66715 Mdot
0 1 .857 r
.53632 .66704 Mdot
.43457 .76051 Mdot
0 1 .88 r
.37741 .66445 Mdot
0 1 .857 r
.44272 .75814 Mdot
.589 0 1 r
.75917 .47145 Mdot
0 1 .882 r
.38785 .70207 Mdot
.363 1 0 r
.73861 .49143 Mdot
0 1 .884 r
.41192 .70152 Mdot
.615 0 1 r
.75935 .47376 Mdot
.779 1 0 r
.46955 .46985 Mdot
0 1 .231 r
.31248 .46951 Mdot
.804 1 0 r
.46951 .47217 Mdot
0 1 .221 r
.31231 .47182 Mdot
0 1 .884 r
.42006 .69904 Mdot
.568 0 1 r
.77321 .46685 Mdot
0 1 .88 r
.39597 .65863 Mdot
.594 0 1 r
.77341 .46917 Mdot
0 1 .882 r
.40647 .69638 Mdot
0 1 .88 r
.4041 .65608 Mdot
.344 1 0 r
.76018 .48443 Mdot
0 1 .206 r
.33084 .46296 Mdot
.027 0 1 r
.4693 .48284 Mdot
0 .684 1 r
.31155 .4825 Mdot
0 1 .196 r
.33069 .46528 Mdot
0 1 .882 r
.41463 .69389 Mdot
0 1 .856 r
.5364 .69377 Mdot
0 1 .879 r
.36517 .65288 Mdot
0 1 .196 r
.33888 .46009 Mdot
.329 1 0 r
.77429 .47985 Mdot
0 1 .186 r
.33874 .46241 Mdot
0 1 .881 r
.37615 .69121 Mdot
0 1 .884 r
.39182 .74406 Mdot
0 .667 1 r
.32999 .47597 Mdot
0 1 .879 r
.38382 .64698 Mdot
.434 1 0 r
.45409 .45248 Mdot
1 .042 0 r
.71825 .45229 Mdot
0 .659 1 r
.33806 .47311 Mdot
0 1 .881 r
.39487 .68544 Mdot
.453 1 0 r
.45404 .4548 Mdot
1 0 0 r
.7184 .45461 Mdot
0 1 .879 r
.39199 .64439 Mdot
0 1 .882 r
.38626 .73902 Mdot
0 1 .885 r
.51387 .64362 Mdot
.41061 .73848 Mdot
0 1 .881 r
.40307 .68291 Mdot
0 1 .643 r
.2909 .44745 Mdot
0 1 .885 r
.41884 .73603 Mdot
1 .374 0 r
.71378 .44626 Mdot
0 1 .64 r
.29071 .44978 Mdot
.441 0 1 r
.83925 .44526 Mdot
0 1 .879 r
.3638 .67974 Mdot
1 .087 0 r
.73997 .44506 Mdot
1 .337 0 r
.71392 .44859 Mdot
0 .216 1 r
.45377 .46552 Mdot
.856 0 1 r
.7191 .46533 Mdot
0 1 .883 r
.5087 .63823 Mdot
.465 0 1 r
.83949 .44759 Mdot
0 1 .883 r
.40509 .7334 Mdot
1 .046 0 r
.74014 .44738 Mdot
.426 0 1 r
.84604 .44304 Mdot
.45 0 1 r
.84629 .44537 Mdot
0 1 .883 r
.41334 .73094 Mdot
0 1 .855 r
.53652 .73083 Mdot
0 1 .634 r
.30941 .44074 Mdot
1 .122 0 r
.75417 .44033 Mdot
0 .974 1 r
.28986 .46051 Mdot
0 1 .877 r
.34531 .6341 Mdot
0 1 .631 r
.30924 .44307 Mdot
1 .414 0 r
.73555 .43898 Mdot
1 .08 0 r
.75435 .44266 Mdot
0 1 .88 r
.38261 .67388 Mdot
.623 0 1 r
.7146 .45932 Mdot
0 1 .631 r
.31752 .4378 Mdot
0 1 .881 r
.37442 .7283 Mdot
1 .377 0 r
.73571 .44131 Mdot
.24 1 0 r
.84063 .45832 Mdot
.825 0 1 r
.74091 .45812 Mdot
0 1 .627 r
.31736 .44013 Mdot
0 1 .88 r
.39085 .67132 Mdot
.229 1 0 r
.84745 .45611 Mdot
0 1 .887 r
.51376 .67056 Mdot
.364 0 1 r
.87247 .4344 Mdot
1 .444 0 r
.74978 .43422 Mdot
1 .56 0 r
.70424 .43343 Mdot
.387 0 1 r
.87274 .43673 Mdot
1 .408 0 r
.74996 .43655 Mdot
0 .968 1 r
.30845 .45381 Mdot
0 1 .877 r
.36411 .62806 Mdot
.801 0 1 r
.75518 .4534 Mdot
1 .526 0 r
.70438 .43576 Mdot
0 1 .882 r
.49769 .62675 Mdot
.595 0 1 r
.73647 .45206 Mdot
0 1 .882 r
.39335 .7226 Mdot
0 .966 1 r
.3166 .45088 Mdot
0 1 .877 r
.37235 .62541 Mdot
0 1 .885 r
.50855 .66521 Mdot
0 1 .882 r
.40163 .72011 Mdot
0 1 .099 r
.43147 .42707 Mdot
0 .978 1 r
.27094 .42705 Mdot
.186 1 0 r
.874 .44749 Mdot
.574 0 1 r
.75077 .44731 Mdot
1 .597 0 r
.72613 .42604 Mdot
0 1 .088 r
.4314 .4294 Mdot
0 .976 1 r
.27074 .42939 Mdot
.492 0 1 r
.70503 .44652 Mdot
1 .563 0 r
.72629 .42838 Mdot
0 1 .88 r
.36191 .71697 Mdot
0 1 .877 r
.34376 .66112 Mdot
.269 0 1 r
.91022 .42206 Mdot
0 1 .875 r
.32835 .61807 Mdot
1 .625 0 r
.74045 .42121 Mdot
.29 0 1 r
.91052 .42439 Mdot
0 .972 1 r
.28959 .42019 Mdot
1 .777 0 r
.69417 .41987 Mdot
0 1 .858 r
.41204 .88267 Mdot
1 .591 0 r
.74062 .42355 Mdot
0 .591 1 r
.43105 .44018 Mdot
0 1 .759 r
.26981 .44016 Mdot
0 .97 1 r
.2894 .42252 Mdot
1 .745 0 r
.6943 .42221 Mdot
1 .326 0 r
.82096 .41809 Mdot
.466 0 1 r
.72702 .43916 Mdot
0 1 .881 r
.48606 .61462 Mdot
.38093 .71119 Mdot
0 .97 1 r
.29776 .41718 Mdot
0 .796 1 r
.26114 .41703 Mdot
0 1 .878 r
.36272 .65513 Mdot
1 .288 0 r
.8212 .42042 Mdot
1 .35 0 r
.82784 .4158 Mdot
0 .967 1 r
.29758 .41952 Mdot
0 .791 1 r
.26093 .41937 Mdot
0 1 .883 r
.49744 .65382 Mdot
0 1 .875 r
.34727 .6119 Mdot
1 .313 0 r
.82808 .41814 Mdot
0 1 .881 r
.38926 .70865 Mdot
0 1 .858 r
.41168 .89356 Mdot
0 1 .878 r
.37102 .6525 Mdot
.118 1 0 r
.91193 .43518 Mdot
0 1 .888 r
.51361 .7079 Mdot
.446 0 1 r
.7414 .43434 Mdot
0 1 .858 r
.4313 .8775 Mdot
1 .81 0 r
.71618 .41237 Mdot
0 1 .755 r
.28853 .43332 Mdot
1 .624 0 r
.81675 .41183 Mdot
.34 0 1 r
.69492 .433 Mdot
0 1 .875 r
.35556 .6092 Mdot
1 .779 0 r
.71633 .41471 Mdot
1 .591 0 r
.81698 .41417 Mdot
0 .783 1 r
.27986 .41009 Mdot
.657 0 1 r
.82228 .43122 Mdot
0 1 .857 r
.43973 .87524 Mdot
1 .646 0 r
.82364 .40952 Mdot
0 1 .753 r
.29674 .43032 Mdot
0 1 .631 r
.25995 .43017 Mdot
0 .778 1 r
.27966 .41243 Mdot
1 .613 0 r
.82388 .41186 Mdot
.64 0 1 r
.82919 .42894 Mdot
1 .835 0 r
.73058 .40747 Mdot
0 1 .886 r
.50834 .70262 Mdot
0 .777 1 r
.28805 .40705 Mdot
0 1 .858 r
.43099 .88841 Mdot
1 .451 0 r
.85457 .40689 Mdot
0 .609 1 r
.25084 .4065 Mdot
1 .805 0 r
.73074 .40981 Mdot
0 .772 1 r
.28786 .40939 Mdot
0 1 .875 r
.32663 .64521 Mdot
1 .414 0 r
.85484 .40924 Mdot
0 .601 1 r
.25062 .40885 Mdot
0 1 .856 r
.7465 .60326 Mdot
.316 0 1 r
.71703 .42552 Mdot
0 1 .597 r
.41089 .40394 Mdot
0 1 .858 r
.43945 .88616 Mdot
.447 0 1 r
.81806 .42498 Mdot
0 1 .593 r
.41079 .40629 Mdot
0 1 .878 r
.34161 .69858 Mdot
0 1 .622 r
.27875 .42325 Mdot
0 1 .882 r
.48571 .64179 Mdot
.432 0 1 r
.82498 .42268 Mdot
1 .734 0 r
.85045 .40056 Mdot
0 .588 1 r
.26963 .39948 Mdot
.298 0 1 r
.7315 .42063 Mdot
1 .702 0 r
.85071 .4029 Mdot
0 1 .618 r
.28698 .42021 Mdot
.569 0 1 r
.85605 .42006 Mdot
0 1 .876 r
.34572 .6391 Mdot
1 .792 0 r
.80778 .3985 Mdot
0 1 .499 r
.2496 .41967 Mdot
0 .58 1 r
.26942 .40183 Mdot
.837 1 0 r
.67782 .39786 Mdot
0 1 .857 r
.76891 .59646 Mdot
1 .761 0 r
.80801 .40084 Mdot
0 .58 1 r
.27785 .39641 Mdot
.863 1 0 r
.67794 .4002 Mdot
1 .812 0 r
.81471 .39616 Mdot
0 .942 1 r
.41037 .41711 Mdot
0 1 .876 r
.35407 .63642 Mdot
0 1 .878 r
.46718 .59492 Mdot
.36078 .69266 Mdot
0 1 .872 r
.30352 .59459 Mdot
0 .572 1 r
.27765 .39875 Mdot
1 .781 0 r
.81495 .39851 Mdot
1 .605 0 r
.89278 .39417 Mdot
0 1 .884 r
.4971 .69137 Mdot
1 .571 0 r
.89308 .39652 Mdot
.369 0 1 r
.85192 .41374 Mdot
0 1 .878 r
.36918 .69006 Mdot
0 1 .857 r
.78357 .59202 Mdot
0 1 .884 r
.38696 .86179 Mdot
0 1 .485 r
.26847 .41267 Mdot
.81 1 0 r
.70003 .39018 Mdot
.329 0 1 r
.80906 .41168 Mdot
.068 0 1 r
.6785 .41104 Mdot
.835 1 0 r
.70017 .39253 Mdot
0 1 .856 r
.74843 .63053 Mdot
0 1 .479 r
.27673 .4096 Mdot
0 1 .872 r
.32262 .58825 Mdot
.315 0 1 r
.81602 .40935 Mdot
1 .871 0 r
.88876 .38775 Mdot
1 .894 0 r
.84167 .38706 Mdot
1 .841 0 r
.88906 .3901 Mdot
0 1 .883 r
.38651 .87278 Mdot
.46 0 1 r
.89444 .40737 Mdot
1 .864 0 r
.84193 .38941 Mdot
0 1 .882 r
.38118 .85698 Mdot
.789 1 0 r
.71455 .38516 Mdot
0 1 .872 r
.33099 .58547 Mdot
0 1 .884 r
.40642 .85646 Mdot
1 .986 0 r
.7983 .38441 Mdot
.814 1 0 r
.71471 .38751 Mdot
0 1 .876 r
.32426 .68286 Mdot
1 .958 0 r
.79852 .38676 Mdot
0 .946 1 r
.39184 .38254 Mdot
.995 1 0 r
.80528 .38203 Mdot
.049 0 1 r
.70082 .40339 Mdot
0 1 .884 r
.41494 .85413 Mdot
0 .943 1 r
.39173 .38489 Mdot
0 1 .856 r
.77104 .62379 Mdot
1 .977 0 r
.8055 .38439 Mdot
0 .169 1 r
.22528 .38038 Mdot
0 1 .883 r
.48522 .67948 Mdot
0 1 .881 r
.38071 .86799 Mdot
.273 0 1 r
.89041 .40096 Mdot
0 1 .879 r
.46666 .62226 Mdot
.48 1 0 r
.66384 .37904 Mdot
0 .155 1 r
.22504 .38273 Mdot
0 1 .872 r
.30156 .62193 Mdot
0 1 .884 r
.40603 .86748 Mdot
.257 0 1 r
.84311 .40028 Mdot
0 1 .882 r
.40068 .85162 Mdot
.5 1 0 r
.66396 .3814 Mdot
.034 0 1 r
.71541 .39838 Mdot
0 1 .876 r
.45104 .57809 Mdot
.34356 .67682 Mdot
0 1 .886 r
.72637 .5779 Mdot
.192 0 1 r
.79954 .39763 Mdot
0 1 .856 r
.78583 .61937 Mdot
0 1 .884 r
.41458 .86515 Mdot
0 1 .882 r
.40923 .84926 Mdot
0 1 .856 r
.53689 .84915 Mdot
.98 1 0 r
.8802 .37406 Mdot
0 1 .737 r
.39122 .39577 Mdot
.179 0 1 r
.80655 .39526 Mdot
0 1 .876 r
.35201 .67417 Mdot
0 .132 1 r
.24424 .37315 Mdot
.921 1 0 r
.8324 .3728 Mdot
1 .992 0 r
.88049 .37642 Mdot
0 1 .882 r
.40027 .86265 Mdot
0 .726 1 r
.38248 .37202 Mdot
0 1 .19 r
.22391 .39361 Mdot
0 1 .881 r
.36888 .84674 Mdot
0 .117 1 r
.24401 .37551 Mdot
0 1 .872 r
.32083 .61563 Mdot
.948 1 0 r
.83265 .37516 Mdot
.458 1 0 r
.68622 .37121 Mdot
0 1 .869 r
.28091 .57321 Mdot
0 .184 1 r
.66447 .39228 Mdot
0 .72 1 r
.38236 .37438 Mdot
0 .116 1 r
.25254 .36999 Mdot
0 1 .884 r
.72173 .57206 Mdot
.478 1 0 r
.68635 .37357 Mdot
0 1 .882 r
.40884 .86031 Mdot
0 1 .856 r
.53693 .8602 Mdot
0 1 .858 r
.85254 .57109 Mdot
0 1 .886 r
.74904 .57089 Mdot
0 .101 1 r
.25231 .37235 Mdot
0 1 .872 r
.32926 .61287 Mdot
0 1 .855 r
.7511 .66835 Mdot
0 1 .858 r
.85964 .56894 Mdot
.442 1 0 r
.70086 .36608 Mdot
0 1 .88 r
.36836 .85779 Mdot
.168 0 1 r
.88183 .38731 Mdot
0 1 .164 r
.24294 .3864 Mdot
.462 1 0 r
.701 .36845 Mdot
0 1 .881 r
.38847 .84129 Mdot
.127 0 1 r
.8338 .38605 Mdot
0 1 .869 r
.30018 .5667 Mdot
0 1 .582 r
.38182 .38527 Mdot
0 1 .886 r
.76387 .56631 Mdot
0 .199 1 r
.68696 .38447 Mdot
0 1 .884 r
.74447 .565 Mdot
.666 1 0 r
.78291 .36153 Mdot
0 1 .153 r
.25128 .38325 Mdot
0 1 .881 r
.39706 .83891 Mdot
0 .501 1 r
.37264 .36096 Mdot
0 1 .869 r
.30861 .56385 Mdot
.689 1 0 r
.78312 .36389 Mdot
0 1 .877 r
.45037 .60556 Mdot
0 1 .887 r
.72815 .60537 Mdot
.807 1 0 r
.87116 .3596 Mdot
0 .491 1 r
.37251 .36333 Mdot
.651 1 0 r
.78995 .35909 Mdot
0 1 .855 r
.77399 .66169 Mdot
0 1 .881 r
.38802 .85237 Mdot
.832 1 0 r
.87144 .36196 Mdot
.674 1 0 r
.79016 .36146 Mdot
0 .21 1 r
.70166 .37935 Mdot
0 1 .879 r
.35587 .83591 Mdot
0 1 .858 r
.88726 .56056 Mdot
0 1 .861 r
.40811 .99975 Mdot
0 1 .883 r
.75933 .56038 Mdot
0 1 .88 r
.46594 .66017 Mdot
0 1 .872 r
.29885 .65985 Mdot
0 1 .883 r
.71185 .55961 Mdot
0 1 .881 r
.39663 .85 Mdot
0 1 .869 r
.27873 .60072 Mdot
0 1 .855 r
.78897 .65732 Mdot
0 1 .885 r
.72347 .59957 Mdot
0 .053 1 r
.78409 .37481 Mdot
0 1 .423 r
.37193 .37425 Mdot
0 1 .857 r
.85544 .59861 Mdot
0 1 .887 r
.75103 .59841 Mdot
0 1 .072 r
.64338 .35149 Mdot
0 1 .879 r
.3553 .84701 Mdot
.047 0 1 r
.87275 .37288 Mdot
0 .063 1 r
.79117 .37238 Mdot
0 1 .06 r
.64348 .35386 Mdot
0 1 .88 r
.37556 .83038 Mdot
.591 1 0 r
.81733 .34963 Mdot
0 1 .857 r
.86261 .59647 Mdot
0 1 .861 r
.42801 .99487 Mdot
0 1 .873 r
.4274 .55344 Mdot
0 1 .867 r
.25998 .55343 Mdot
0 1 .873 r
.31833 .65363 Mdot
.613 1 0 r
.81757 .352 Mdot
0 1 .882 r
.73471 .55245 Mdot
0 1 .869 r
.29816 .59426 Mdot
0 1 .88 r
.38419 .82795 Mdot
0 1 .887 r
.766 .59386 Mdot
0 1 .861 r
.43673 .99272 Mdot
0 1 .887 r
.51314 .82723 Mdot
0 1 .873 r
.32687 .6509 Mdot
0 1 .885 r
.74642 .59256 Mdot
0 1 .869 r
.30667 .59143 Mdot
0 1 .859 r
.92674 .54858 Mdot
0 1 .88 r
.37506 .84151 Mdot
0 1 .085 r
.666 .34342 Mdot
0 .572 1 r
.64392 .3648 Mdot
0 1 .882 r
.74967 .54776 Mdot
.345 1 0 r
.76975 .34196 Mdot
0 1 .073 r
.66612 .3458 Mdot
0 1 .867 r
.27939 .54676 Mdot
0 1 .881 r
.7014 .54645 Mdot
0 .106 1 r
.81868 .36294 Mdot
.363 1 0 r
.76995 .34434 Mdot
0 1 .88 r
.38372 .8391 Mdot
0 1 .857 r
.8905 .58815 Mdot
0 1 .885 r
.76142 .58798 Mdot
.333 1 0 r
.77684 .33948 Mdot
0 1 .886 r
.5131 .83838 Mdot
0 1 .884 r
.83367 .54472 Mdot
.71352 .58722 Mdot
0 1 .885 r
.50767 .82218 Mdot
0 1 .867 r
.2879 .54384 Mdot
0 1 .094 r
.6808 .33815 Mdot
0 1 .866 r
.2497 .5437 Mdot
.351 1 0 r
.77705 .34185 Mdot
0 1 .863 r
.4069 1.03567 Mdot
0 1 .877 r
.44944 .64366 Mdot
0 1 .889 r
.73061 .64348 Mdot
0 1 .884 r
.84086 .5425 Mdot
0 1 .083 r
.68093 .34052 Mdot
.498 1 0 r
.85646 .3361 Mdot
0 .581 1 r
.66665 .35675 Mdot
.518 1 0 r
.85674 .33847 Mdot
0 1 .877 r
.33473 .81831 Mdot
.029 0 1 r
.34821 .33352 Mdot
0 .279 1 r
.77088 .35529 Mdot
0 1 .881 r
.7244 .53917 Mdot
0 1 .884 r
.50761 .83335 Mdot
0 1 .882 r
.82932 .53864 Mdot
.047 0 1 r
.34806 .33589 Mdot
0 1 .873 r
.42651 .58109 Mdot
0 1 .867 r
.25759 .58108 Mdot
0 1 .869 r
.2757 .63888 Mdot
0 1 .884 r
.73659 .58011 Mdot
0 .287 1 r
.77801 .35281 Mdot
0 1 .865 r
.26918 .53696 Mdot
0 1 .862 r
.42701 1.03087 Mdot
0 1 .886 r
.72589 .63775 Mdot
0 1 .882 r
.83653 .5364 Mdot
.285 1 0 r
.80444 .32982 Mdot
0 .588 1 r
.68152 .35148 Mdot
0 1 .856 r
.85948 .6368 Mdot
0 1 .888 r
.75378 .6366 Mdot
.302 1 0 r
.80467 .33219 Mdot
0 1 .881 r
.73944 .53441 Mdot
0 1 .877 r
.33409 .8295 Mdot
0 .171 1 r
.85801 .34944 Mdot
0 1 .865 r
.27771 .534 Mdot
0 1 .883 r
.86882 .53385 Mdot
0 1 .878 r
.38209 .98 Mdot
0 1 .862 r
.43581 1.02877 Mdot
0 1 .864 r
.23888 .53347 Mdot
0 1 .587 r
.62473 .32639 Mdot
0 1 .858 r
.93036 .57626 Mdot
0 1 .856 r
.86674 .63468 Mdot
0 1 .878 r
.35459 .81264 Mdot
0 1 .863 r
.40617 1.05735 Mdot
0 1 .443 r
.40617 1.05746 Mdot
0 1 .883 r
.75168 .57545 Mdot
0 1 .583 r
.62482 .32877 Mdot
0 1 .05 r
.34738 .34687 Mdot
0 1 .883 r
.49601 .81141 Mdot
0 1 .867 r
.27716 .57446 Mdot
0 1 .882 r
.70299 .57415 Mdot
0 1 .87 r
.40587 .53098 Mdot
0 1 .869 r
.29535 .63249 Mdot
0 1 .888 r
.76894 .6321 Mdot
.269 0 1 r
.40598 1.06315 Mdot
0 1 .878 r
.36329 .81016 Mdot
0 1 .885 r
.83646 .57244 Mdot
0 1 .886 r
.74912 .63082 Mdot
0 1 .867 r
.28574 .57156 Mdot
0 .321 1 r
.80575 .34317 Mdot
0 1 .865 r
.24719 .57142 Mdot
0 1 .869 r
.30396 .62969 Mdot
0 1 .882 r
.86457 .52769 Mdot
0 1 .877 r
.37609 .97545 Mdot
.35401 .82386 Mdot
0 1 .885 r
.84371 .57023 Mdot
0 1 .879 r
.40221 .97496 Mdot
0 1 .864 r
.25844 .52665 Mdot
0 1 .863 r
.4264 1.05261 Mdot
0 1 .427 r
.42639 1.05272 Mdot
0 1 .592 r
.64758 .3181 Mdot
0 1 .883 r
.49591 .82263 Mdot
0 1 .881 r
.82006 .52569 Mdot
0 .935 1 r
.6252 .33975 Mdot
0 1 .879 r
.68443 .52507 Mdot
0 1 .588 r
.64768 .32049 Mdot
0 1 .857 r
.89499 .62646 Mdot
0 1 .878 r
.36274 .82139 Mdot
0 1 .886 r
.76432 .62629 Mdot
.211 1 0 r
.8439 .316 Mdot
0 1 .879 r
.41102 .97274 Mdot
0 1 .864 r
.267 .52366 Mdot
0 1 .882 r
.7262 .56693 Mdot
0 1 .885 r
.71583 .62553 Mdot
0 1 .881 r
.82731 .52342 Mdot
.323 0 1 r
.42623 1.05842 Mdot
0 1 .883 r
.83208 .5664 Mdot
0 1 .863 r
.43526 1.05053 Mdot
0 1 .42 r
.43526 1.05064 Mdot
.228 1 0 r
.84416 .31839 Mdot
0 1 .875 r
.31666 .80327 Mdot
0 1 .152 r
.75046 .3133 Mdot
0 1 .882 r
.9088 .52149 Mdot
0 1 .865 r
.26684 .56473 Mdot
0 1 .595 r
.66252 .31269 Mdot
0 1 .883 r
.83935 .56418 Mdot
0 1 .877 r
.39626 .97036 Mdot
.345 0 1 r
.43511 1.05635 Mdot
0 1 .141 r
.75065 .31568 Mdot
0 1 .591 r
.66263 .31507 Mdot
0 1 .159 r
.75764 .31074 Mdot
0 1 .882 r
.74138 .5622 Mdot
0 1 .865 r
.27545 .5618 Mdot
0 1 .885 r
.87194 .56165 Mdot
0 1 .882 r
.48367 .80003 Mdot
0 1 .148 r
.75783 .31313 Mdot
0 1 .864 r
.23626 .56127 Mdot
0 .938 1 r
.64815 .33149 Mdot
0 1 .877 r
.40509 .96814 Mdot
0 1 .86 r
.53727 .96803 Mdot
0 1 .879 r
.70764 .51761 Mdot
0 1 .873 r
.42528 .61947 Mdot
0 1 .867 r
.25426 .61946 Mdot
0 1 .876 r
.38059 1.01628 Mdot
0 1 .885 r
.7392 .6185 Mdot
0 1 .875 r
.31594 .81452 Mdot
0 .373 1 r
.8454 .32939 Mdot
0 1 .87 r
.40477 .5588 Mdot
0 1 .881 r
.90467 .51524 Mdot
0 1 .876 r
.33665 .79748 Mdot
0 1 .881 r
.85552 .51458 Mdot
0 1 .876 r
.36331 .96574 Mdot
0 .628 1 r
.75152 .32669 Mdot
0 .94 1 r
.66316 .32608 Mdot
0 1 .879 r
.72283 .51272 Mdot
0 .94 1 r
.60746 .30314 Mdot
0 1 .882 r
.48353 .8113 Mdot
0 1 .88 r
.81026 .51199 Mdot
0 1 .883 r
.86767 .55553 Mdot
0 1 .857 r
.93538 .6147 Mdot
0 1 .876 r
.34542 .79494 Mdot
0 .633 1 r
.75874 .32414 Mdot
0 .937 1 r
.60753 .30552 Mdot
0 1 .884 r
.75448 .61389 Mdot
0 1 .864 r
.25599 .5545 Mdot
0 1 .187 r
.78556 .30078 Mdot
0 1 .867 r
.38592 .51018 Mdot
.27406 .61291 Mdot
0 1 .882 r
.82275 .55355 Mdot
0 1 .875 r
.37452 1.01181 Mdot
0 1 .883 r
.70518 .61261 Mdot
0 1 .88 r
.81756 .50968 Mdot
.68588 .55293 Mdot
0 1 .876 r
.40091 1.01133 Mdot
0 1 .177 r
.78578 .30317 Mdot
0 1 .875 r
.33601 .80876 Mdot
0 1 .861 r
.21202 .50807 Mdot
0 1 .887 r
.84032 .61091 Mdot
0 1 .864 r
.26463 .55153 Mdot
0 1 .882 r
.83007 .55129 Mdot
0 1 .877 r
.38358 .96058 Mdot
0 1 .867 r
.28274 .61005 Mdot
0 1 .865 r
.24371 .60991 Mdot
0 1 .876 r
.66991 .50678 Mdot
.40982 1.00916 Mdot
.3448 .80623 Mdot
0 1 .884 r
.91231 .54937 Mdot
0 1 .886 r
.84767 .60873 Mdot
0 1 .875 r
.37969 1.03819 Mdot
0 .447 1 r
.37968 1.0383 Mdot
0 1 .856 r
.75963 .78936 Mdot
0 .943 1 r
.63051 .29465 Mdot
0 1 .877 r
.39246 .95832 Mdot
0 1 .732 r
.60785 .31655 Mdot
0 .94 1 r
.63059 .29704 Mdot
0 1 .875 r
.3949 1.00682 Mdot
1 .605 0 r
.37945 1.04405 Mdot
0 .712 1 r
.59897 .2917 Mdot
0 .653 1 r
.78679 .31421 Mdot
0 1 .883 r
.72869 .60546 Mdot
0 1 .88 r
.89585 .50193 Mdot
0 1 .879 r
.70931 .54552 Mdot
0 1 .884 r
.8359 .60494 Mdot
0 1 .861 r
.23176 .50105 Mdot
0 .706 1 r
.59903 .29409 Mdot
0 1 .879 r
.84594 .5007 Mdot
0 1 .875 r
.34979 .95548 Mdot
0 .945 1 r
.64558 .2891 Mdot
0 1 .865 r
.37611 .49994 Mdot
.26359 .60329 Mdot
.966 0 1 r
.13564 .28873 Mdot
0 1 .875 r
.40382 1.00464 Mdot
0 1 .862 r
.53738 1.00453 Mdot
0 1 .882 r
.90815 .54317 Mdot
0 1 .856 r
.76043 .80068 Mdot
0 1 .884 r
.84327 .60274 Mdot
0 1 .876 r
.69331 .49915 Mdot
0 1 .874 r
.37358 1.03377 Mdot
0 .527 1 r
.37357 1.03388 Mdot
0 1 .881 r
.85855 .54251 Mdot
0 .942 1 r
.64568 .29149 Mdot
0 1 .615 r
.73288 .28717 Mdot
0 1 .875 r
.40013 1.0333 Mdot
.998 0 1 r
.13531 .29113 Mdot
0 .42 1 r
.40013 1.03341 Mdot
0 1 .861 r
.2404 .49797 Mdot
0 1 .231 r
.82548 .28655 Mdot
0 1 .856 r
.78344 .78297 Mdot
0 1 .883 r
.74408 .60079 Mdot
0 1 .612 r
.73306 .28956 Mdot
0 1 .879 r
.72464 .54067 Mdot
0 1 .865 r
.2723 .60039 Mdot
0 1 .734 r
.631 .30808 Mdot
0 1 .886 r
.87628 .60024 Mdot
0 1 .875 r
.3616 1.00229 Mdot
0 1 .221 r
.82573 .28894 Mdot
0 1 .864 r
.23263 .59987 Mdot
1 .871 0 r
.37333 1.03965 Mdot
0 1 .618 r
.74013 .28454 Mdot
0 1 .881 r
.81288 .53995 Mdot
0 1 .879 r
.46363 .78152 Mdot
0 1 .872 r
.29016 .78121 Mdot
1 .518 0 r
.39992 1.03918 Mdot
0 1 .875 r
.40909 1.03115 Mdot
0 .409 1 r
.40909 1.03126 Mdot
0 .719 1 r
.62211 .28311 Mdot
0 1 .614 r
.74032 .28693 Mdot
0 1 .876 r
.70861 .49417 Mdot
0 1 .866 r
.38462 .53814 Mdot
0 1 .572 r
.59931 .30514 Mdot
0 1 .881 r
.82025 .53765 Mdot
0 1 .876 r
.37017 .95024 Mdot
0 1 .87 r
.40324 .59742 Mdot
0 .713 1 r
.62219 .28551 Mdot
1 0 .953 r
.15516 .28078 Mdot
0 1 .856 r
.79902 .77879 Mdot
1 .481 0 r
.4089 1.03704 Mdot
0 1 .861 r
.20911 .53605 Mdot
0 1 .874 r
.39408 1.02884 Mdot
0 .504 1 r
.39407 1.02895 Mdot
0 1 .736 r
.64615 .30254 Mdot
0 .478 1 r
.59004 .27968 Mdot
0 1 .856 r
.78433 .79432 Mdot
.609 1 0 r
.13377 .30218 Mdot
1 0 .92 r
.15485 .28318 Mdot
0 1 .877 r
.67124 .53477 Mdot
0 1 .876 r
.3791 .94794 Mdot
0 .469 1 r
.59009 .28207 Mdot
0 1 .884 r
.87196 .59419 Mdot
0 .955 1 r
.73387 .30062 Mdot
0 1 .879 r
.46341 .79287 Mdot
0 1 .878 r
.79434 .48973 Mdot
0 .724 1 r
.63725 .2775 Mdot
0 1 .88 r
.51267 .94725 Mdot
1 0 .919 r
.16372 .2773 Mdot
0 1 .872 r
.28935 .79256 Mdot
0 .684 1 r
.82691 .3 Mdot
0 1 .875 r
.38208 .99722 Mdot
0 1 .864 r
.36579 .48918 Mdot
.25258 .59316 Mdot
1 .794 0 r
.39386 1.03474 Mdot
0 1 .874 r
.40306 1.02668 Mdot
0 .494 1 r
.40305 1.02679 Mdot
0 1 .863 r
.53745 1.02658 Mdot
0 1 .356 r
.53745 1.02669 Mdot
0 1 .872 r
.31036 .77524 Mdot
0 .718 1 r
.63735 .27989 Mdot
0 1 .883 r
.82648 .59222 Mdot
1 0 .886 r
.16341 .2797 Mdot
0 1 .879 r
.88653 .48785 Mdot
0 .956 1 r
.74116 .298 Mdot
0 1 .88 r
.68789 .59161 Mdot
0 1 .878 r
.8017 .48736 Mdot
0 1 .628 r
.76834 .27431 Mdot
0 1 .856 r
.79996 .79015 Mdot
0 1 .577 r
.62257 .29657 Mdot
0 1 .875 r
.39105 .995 Mdot
0 1 .864 r
.26133 .59023 Mdot
1 .761 0 r
.40285 1.03258 Mdot
0 1 .881 r
.89928 .52995 Mdot
0 1 .883 r
.8339 .58999 Mdot
.557 0 1 r
.53747 1.03248 Mdot
0 1 .873 r
.36056 1.02436 Mdot
0 1 .624 r
.76855 .27671 Mdot
0 .573 1 r
.36056 1.02447 Mdot
0 1 .872 r
.31921 .77263 Mdot
0 1 .861 r
.22902 .52907 Mdot
0 1 .88 r
.84891 .52873 Mdot
.666 1 0 r
.15338 .29425 Mdot
0 .489 1 r
.61328 .27098 Mdot
0 1 .865 r
.37471 .52798 Mdot
0 1 .884 r
.91719 .58809 Mdot
0 1 .408 r
.59034 .29314 Mdot
0 1 .879 r
.507 .94246 Mdot
0 1 .877 r
.69486 .5272 Mdot
.98 1 0 r
.36029 1.03027 Mdot
0 .479 1 r
.61336 .27338 Mdot
1 0 .661 r
.11661 .26928 Mdot
0 1 .872 r
.30961 .78662 Mdot
0 1 .874 r
.34792 .99221 Mdot
0 1 .861 r
.23774 .52602 Mdot
0 1 .58 r
.63779 .29097 Mdot
.69 1 0 r
.16197 .29077 Mdot
1 0 .624 r
.11626 .27168 Mdot
0 1 .873 r
.64862 .47996 Mdot
0 1 .88 r
.71162 .58428 Mdot
0 1 .872 r
.31849 .78402 Mdot
0 .497 1 r
.62849 .2653 Mdot
0 .963 1 r
.76951 .28779 Mdot
0 1 .874 r
.38117 1.01935 Mdot
0 1 .877 r
.83037 .47814 Mdot
0 1 .874 r
.32781 .93879 Mdot
0 .551 1 r
.38116 1.01946 Mdot
0 1 .877 r
.71031 .52224 Mdot
0 .487 1 r
.62858 .2677 Mdot
0 1 .882 r
.91299 .58195 Mdot
0 1 .877 r
.44648 .76569 Mdot
0 .959 1 r
.71659 .26295 Mdot
0 1 .888 r
.7385 .76551 Mdot
0 1 .882 r
.86276 .5813 Mdot
0 1 .415 r
.6137 .28447 Mdot
0 .956 1 r
.71675 .26535 Mdot
1 .949 0 r
.38092 1.02528 Mdot
1 0 .566 r
.13625 .26117 Mdot
.269 0 1 r
.10855 .26104 Mdot
0 1 .874 r
.39019 1.01716 Mdot
.36851 .98706 Mdot
0 .542 1 r
.39019 1.01727 Mdot
0 1 .88 r
.72716 .57948 Mdot
0 .961 1 r
.72391 .26026 Mdot
.872 1 0 r
.11463 .28277 Mdot
0 1 .643 r
.80868 .25968 Mdot
0 1 .882 r
.81652 .57876 Mdot
1 0 .527 r
.13591 .26357 Mdot
.29 0 1 r
.10819 .26345 Mdot
0 1 .878 r
.79684 .51784 Mdot
0 .958 1 r
.72408 .26266 Mdot
0 1 .863 r
.36429 .51729 Mdot
0 1 .64 r
.80892 .26209 Mdot
0 1 .873 r
.67228 .4721 Mdot
1 .919 0 r
.38997 1.02309 Mdot
1 0 .526 r
.14485 .25761 Mdot
0 1 .866 r
.38282 .57698 Mdot
0 1 .874 r
.37753 .98481 Mdot
0 1 .869 r
.266 .7611 Mdot
0 1 .879 r
.8899 .51597 Mdot
0 1 .882 r
.82398 .57649 Mdot
0 1 .873 r
.34679 1.0144 Mdot
0 .625 1 r
.34679 1.01451 Mdot
0 1 .878 r
.80428 .51548 Mdot
0 1 .874 r
.34837 .93342 Mdot
0 1 .878 r
.51253 .98413 Mdot
0 1 .877 r
.4462 .77711 Mdot
0 1 .876 r
.78071 .47068 Mdot
0 1 .887 r
.73923 .77693 Mdot
1 0 .486 r
.14452 .26002 Mdot
0 1 .42 r
.62899 .27879 Mdot
0 1 .885 r
.73362 .76001 Mdot
0 1 .861 r
.20508 .57491 Mdot
0 1 .878 r
.49491 .93225 Mdot
0 1 .857 r
.8724 .7591 Mdot
0 1 .887 r
.7626 .75891 Mdot
0 1 .878 r
.67308 .57364 Mdot
0 1 .876 r
.78813 .46826 Mdot
0 1 .746 r
.71751 .27645 Mdot
.807 1 0 r
.34649 1.02034 Mdot
.323 0 1 r
.12824 .25286 Mdot
0 1 .874 r
.35737 .93106 Mdot
0 1 .873 r
.68777 .46696 Mdot
0 1 .857 r
.87996 .75707 Mdot
.939 1 0 r
.13435 .27467 Mdot
.118 1 0 r
.10652 .27455 Mdot
.344 0 1 r
.1279 .25527 Mdot
0 .754 1 r
.70857 .25104 Mdot
0 1 .747 r
.72487 .27376 Mdot
0 1 .869 r
.26509 .77253 Mdot
0 .974 1 r
.81005 .27319 Mdot
.07 0 1 r
.56784 .24979 Mdot
0 .967 1 r
.75238 .24977 Mdot
0 1 .877 r
.87137 .46496 Mdot
0 .749 1 r
.70873 .25344 Mdot
.345 0 1 r
.13687 .24928 Mdot
0 1 .877 r
.50679 .97943 Mdot
0 1 .869 r
.28637 .75496 Mdot
0 1 .885 r
.73434 .77145 Mdot
0 1 .873 r
.36751 1.00931 Mdot
0 .758 1 r
.71592 .24831 Mdot
0 1 .887 r
.77838 .75459 Mdot
0 .605 1 r
.3675 1.00942 Mdot
.088 0 1 r
.56787 .2522 Mdot
0 .965 1 r
.75257 .25218 Mdot
0 1 .873 r
.64977 .50813 Mdot
.967 1 0 r
.143 .27113 Mdot
0 1 .881 r
.90405 .56887 Mdot
.368 0 1 r
.13653 .25169 Mdot
0 1 .857 r
.8736 .77054 Mdot
0 1 .887 r
.76343 .77036 Mdot
0 1 .86 r
.34015 .46244 Mdot
.22522 .568 Mdot
0 1 .885 r
.7578 .75336 Mdot
0 .752 1 r
.71609 .25072 Mdot
0 1 .881 r
.85303 .56766 Mdot
0 1 .878 r
.83323 .50633 Mdot
0 1 .865 r
.37277 .56692 Mdot
0 1 .869 r
.2953 .75228 Mdot
.871 1 0 r
.36724 1.01526 Mdot
0 1 .873 r
.37658 1.00708 Mdot
0 .597 1 r
.37658 1.00719 Mdot
0 1 .857 r
.88119 .76852 Mdot
0 1 .878 r
.69701 .56614 Mdot
0 1 .876 r
.51244 1.00641 Mdot
0 .306 1 r
.51244 1.00653 Mdot
0 1 .873 r
.32569 .97583 Mdot
0 1 .86 r
.23405 .56498 Mdot
.156 1 0 r
.1263 .26638 Mdot
0 1 .875 r
.81704 .45884 Mdot
0 1 .873 r
.309 .92451 Mdot
0 1 .869 r
.28553 .76643 Mdot
.899 1 0 r
.37633 1.01304 Mdot
0 1 .886 r
.77926 .76605 Mdot
0 1 .602 r
.70946 .26456 Mdot
0 1 .857 r
.90938 .74917 Mdot
.051 0 1 r
.59134 .24083 Mdot
0 1 .885 r
.77362 .749 Mdot
1 .139 0 r
.51241 1.01238 Mdot
0 1 .021 r
.56804 .26332 Mdot
0 1 .752 r
.75348 .2633 Mdot
0 1 .884 r
.72323 .74828 Mdot
.75861 .76483 Mdot
.172 1 0 r
.13496 .26281 Mdot
0 1 .87 r
.6292 .4555 Mdot
.069 0 1 r
.5914 .24325 Mdot
0 1 .873 r
.67366 .50032 Mdot
0 1 .877 r
.71267 .56124 Mdot
0 1 .604 r
.71685 .26184 Mdot
0 .544 1 r
.70014 .2385 Mdot
0 1 .877 r
.48211 .92144 Mdot
0 1 .869 r
.29449 .76375 Mdot
0 .773 1 r
.74452 .2377 Mdot
0 1 .876 r
.78311 .49891 Mdot
1 0 .601 r
.2624 .2371 Mdot
0 .535 1 r
.7003 .24092 Mdot
0 1 .875 r
.50667 1.00176 Mdot
0 .403 1 r
.50667 1.00187 Mdot
0 .767 1 r
.74471 .24011 Mdot
0 .55 1 r
.70753 .23574 Mdot
1 0 .563 r
.26216 .23951 Mdot
.037 0 1 r
.60672 .23497 Mdot
0 1 .873 r
.34645 .97054 Mdot
0 .978 1 r
.7931 .23478 Mdot
0 1 .876 r
.79061 .49651 Mdot
0 1 .873 r
.3297 .91902 Mdot
0 1 .858 r
.91072 .76066 Mdot
0 .541 1 r
.70769 .23816 Mdot
0 1 .884 r
.77448 .76049 Mdot
0 1 .879 r
.80032 .55688 Mdot
.054 0 1 r
.60679 .23738 Mdot
0 1 .876 r
.49457 .96939 Mdot
0 1 .873 r
.6893 .49521 Mdot
0 .976 1 r
.79334 .23719 Mdot
0 1 .863 r
.36221 .55634 Mdot
0 1 .883 r
.72392 .75977 Mdot
1 .459 0 r
.50663 1.00774 Mdot
0 1 .873 r
.42131 .74246 Mdot
0 1 .867 r
.24359 .74245 Mdot
0 1 .88 r
.89458 .55503 Mdot
0 1 .873 r
.35555 .96823 Mdot
0 1 .035 r
.59166 .25438 Mdot
0 1 .873 r
.33877 .91661 Mdot
0 1 .877 r
.87464 .49323 Mdot
0 1 .884 r
.74755 .74152 Mdot
0 1 .879 r
.80786 .55455 Mdot
0 1 .872 r
.3244 .9982 Mdot
0 .719 1 r
.32439 .99831 Mdot
0 1 .87 r
.65311 .44742 Mdot
0 1 .454 r
.701 .25205 Mdot
0 1 .86 r
.33839 .49073 Mdot
0 1 .615 r
.7456 .25125 Mdot
0 1 .875 r
.8584 .44536 Mdot
.914 1 0 r
.2611 .25065 Mdot
.498 1 0 r
.32406 1.00419 Mdot
0 1 .858 r
.95146 .73787 Mdot
0 1 .458 r
.70842 .2493 Mdot
0 1 .045 r
.60712 .24853 Mdot
0 .573 1 r
.73626 .225 Mdot
0 1 .873 r
.42094 .75398 Mdot
0 1 .867 r
.24259 .75396 Mdot
0 1 .883 r
.76346 .7371 Mdot
0 1 .759 r
.79442 .24833 Mdot
0 1 .872 r
.76071 .44273 Mdot
0 1 .87 r
.66875 .44213 Mdot
0 1 .876 r
.81981 .48715 Mdot
0 1 .883 r
.74833 .75304 Mdot
0 1 .867 r
.26413 .73616 Mdot
0 1 .882 r
.71223 .73587 Mdot
0 .564 1 r
.73645 .22741 Mdot
0 1 .86 r
.76824 .91131 Mdot
0 .796 1 r
.78544 .22252 Mdot
0 1 .874 r
.65137 .54727 Mdot
0 1 .872 r
.76823 .44023 Mdot
.34529 .99298 Mdot
0 .703 1 r
.34529 .99309 Mdot
0 1 .872 r
.30665 .9618 Mdot
0 1 .886 r
.8527 .73423 Mdot
0 .791 1 r
.78567 .22494 Mdot
0 1 .875 r
.49436 .99184 Mdot
0 .457 1 r
.49436 .99195 Mdot
0 1 .87 r
.63018 .48383 Mdot
0 1 .879 r
.83721 .54549 Mdot
0 1 .867 r
.27313 .7334 Mdot
0 1 .865 r
.23257 .73327 Mdot
0 1 .858 r
.95297 .74941 Mdot
.55 1 0 r
.34498 .99898 Mdot
0 1 .872 r
.35445 .99069 Mdot
0 1 .885 r
.86035 .73214 Mdot
0 .696 1 r
.35445 .9908 Mdot
0 1 .883 r
.7643 .74864 Mdot
1 .639 0 r
.49431 .99785 Mdot
1 0 .151 r
.24415 .2166 Mdot
0 1 .867 r
.2632 .7477 Mdot
0 1 .875 r
.48163 .95878 Mdot
0 1 .882 r
.71289 .74741 Mdot
0 1 .474 r
.7373 .23857 Mdot
1 0 .107 r
.2439 .21902 Mdot
.572 1 0 r
.35416 .9967 Mdot
0 1 .882 r
.7367 .729 Mdot
0 1 .885 r
.85385 .74578 Mdot
0 1 .866 r
.61119 .43281 Mdot
0 1 .86 r
.79297 .90523 Mdot
0 1 .883 r
.84815 .7285 Mdot
0 1 .874 r
.67558 .53954 Mdot
0 1 .867 r
.27223 .74496 Mdot
0 1 .872 r
.32756 .9564 Mdot
0 1 .865 r
.23152 .74482 Mdot
0 1 .631 r
.78672 .2361 Mdot
0 1 .875 r
.4613 .90386 Mdot
0 1 .865 r
.25318 .7269 Mdot
0 1 .877 r
.78645 .53814 Mdot
0 1 .872 r
.79749 .43051 Mdot
0 1 .871 r
.2814 .90356 Mdot
0 1 .885 r
.86154 .74369 Mdot
0 1 .87 r
.65432 .4758 Mdot
0 1 .883 r
.85583 .72638 Mdot
0 .609 1 r
.77737 .20963 Mdot
0 1 .872 r
.33673 .95403 Mdot
0 1 .875 r
.86158 .47376 Mdot
0 1 .877 r
.79405 .53576 Mdot
0 1 .871 r
.30522 .98433 Mdot
0 1 .882 r
.75272 .7245 Mdot
0 .805 1 r
.30522 .98444 Mdot
.557 0 1 r
.23642 .20792 Mdot
0 .601 1 r
.7776 .21205 Mdot
0 1 .86 r
.80915 .90126 Mdot
0 1 .865 r
.26222 .72411 Mdot
0 .05 1 r
.67918 .20735 Mdot
0 1 .885 r
.89017 .72397 Mdot
0 1 .873 r
.69142 .53448 Mdot
0 1 .864 r
.22097 .72361 Mdot
0 1 .882 r
.73745 .74057 Mdot
1 .77 0 r
.24275 .23019 Mdot
.583 0 1 r
.23617 .21034 Mdot
0 1 .883 r
.8493 .74007 Mdot
0 .034 1 r
.67932 .20977 Mdot
0 1 .873 r
.76296 .47114 Mdot
0 1 .87 r
.67012 .47055 Mdot
0 1 .878 r
.87919 .53252 Mdot
0 .061 1 r
.68665 .2045 Mdot
.211 1 0 r
.30484 .99036 Mdot
0 1 .867 r
.63532 .42452 Mdot
0 1 .865 r
.25221 .73848 Mdot
0 1 .87 r
.39835 .72126 Mdot
0 1 .883 r
.85701 .73796 Mdot
0 1 .874 r
.48134 .98134 Mdot
0 .52 1 r
.48134 .98145 Mdot
0 .045 1 r
.6868 .20692 Mdot
0 1 .873 r
.77056 .46866 Mdot
0 1 .871 r
.30231 .89789 Mdot
0 1 .859 r
.33593 .53004 Mdot
0 1 .865 r
.60233 .42164 Mdot
0 1 .882 r
.75353 .73609 Mdot
0 1 .861 r
.77088 .94882 Mdot
0 1 .865 r
.26128 .7357 Mdot
0 1 .884 r
.89147 .73556 Mdot
1 .848 0 r
.48127 .98738 Mdot
0 1 .864 r
.21988 .73521 Mdot
0 1 .883 r
.88573 .71815 Mdot
0 1 .499 r
.77863 .22323 Mdot
0 1 .871 r
.32627 .97899 Mdot
0 .793 1 r
.32626 .9791 Mdot
0 1 .871 r
.31148 .8954 Mdot
0 1 .867 r
.65111 .4191 Mdot
0 1 .864 r
.24167 .71717 Mdot
0 1 .854 r
.11754 .41874 Mdot
.322 1 0 r
.23498 .22152 Mdot
0 1 .876 r
.82365 .52649 Mdot
0 1 .106 r
.67995 .22095 Mdot
0 1 .882 r
.83846 .71626 Mdot
0 1 .87 r
.39789 .73286 Mdot
0 1 .869 r
.74246 .41721 Mdot
0 1 .88 r
.69434 .71567 Mdot
.253 1 0 r
.32592 .98504 Mdot
0 1 .871 r
.33549 .97665 Mdot
0 1 .872 r
.83935 .41661 Mdot
0 .787 1 r
.33549 .97676 Mdot
0 1 .864 r
.25074 .71434 Mdot
0 1 .114 r
.68745 .21811 Mdot
0 1 .866 r
.61201 .46128 Mdot
0 1 .882 r
.84619 .71411 Mdot
0 .103 1 r
.71572 .19342 Mdot
0 1 .87 r
.63154 .5232 Mdot
0 1 .869 r
.75006 .41465 Mdot
0 1 .882 r
.88702 .72977 Mdot
.27 1 0 r
.33516 .9827 Mdot
0 1 .865 r
.62656 .41325 Mdot
0 1 .884 r
.93282 .71229 Mdot
0 .088 1 r
.71589 .19585 Mdot
0 1 .872 r
.80011 .459 Mdot
0 1 .861 r
.7959 .94286 Mdot
0 1 .864 r
.24065 .72879 Mdot
0 1 .882 r
.83958 .72789 Mdot
0 1 .854 r
.13789 .41097 Mdot
0 1 .874 r
.46059 .9415 Mdot
0 1 .879 r
.69494 .7273 Mdot
0 1 .874 r
.44348 .8888 Mdot
0 1 .87 r
.27871 .94121 Mdot
0 1 .881 r
.74645 .88863 Mdot
0 1 .863 r
.593 .40989 Mdot
0 1 .862 r
.77248 .9715 Mdot
0 1 .338 r
.77249 .97161 Mdot
0 1 .864 r
.24975 .72597 Mdot
0 1 .879 r
.71905 .70862 Mdot
0 1 .882 r
.84734 .72575 Mdot
0 1 .865 r
.64242 .40776 Mdot
0 1 .853 r
.14682 .40757 Mdot
0 1 .862 r
.81227 .93895 Mdot
0 1 .866 r
.63637 .45305 Mdot
0 1 .883 r
.93428 .72393 Mdot
0 1 .87 r
.65601 .51525 Mdot
0 1 .882 r
.92851 .70638 Mdot
.617 0 1 r
.77291 .97757 Mdot
0 1 .144 r
.71668 .20705 Mdot
0 1 .882 r
.87628 .70575 Mdot
0 1 .869 r
.77964 .40465 Mdot
.2562 .88443 Mdot
0 1 .876 r
.866 .51323 Mdot
0 1 .865 r
.60307 .45018 Mdot
0 1 .879 r
.73523 .704 Mdot
.74142 .88339 Mdot
0 1 .87 r
.29984 .93563 Mdot
0 1 .881 r
.82819 .70331 Mdot
0 1 .879 r
.71975 .72027 Mdot
0 1 .861 r
.88543 .88253 Mdot
0 1 .881 r
.77151 .88235 Mdot
0 1 .864 r
.61735 .4014 Mdot
0 1 .873 r
.76609 .51063 Mdot
0 1 .866 r
.65232 .44766 Mdot
0 1 .853 r
.11357 .44731 Mdot
0 1 .862 r
.79767 .9656 Mdot
0 1 .87 r
.67202 .51004 Mdot
0 1 .346 r
.79768 .96571 Mdot
0 .169 1 r
.75732 .17756 Mdot
0 1 .866 r
.37704 .70159 Mdot
0 1 .881 r
.83598 .70113 Mdot
0 1 .852 r
.09743 .39973 Mdot
0 1 .861 r
.8933 .88059 Mdot
0 1 .87 r
.74457 .44578 Mdot
0 1 .881 r
.92997 .71804 Mdot
0 1 .87 r
.3091 .93319 Mdot
0 .155 1 r
.75753 .17999 Mdot
0 1 .873 r
.46016 .96426 Mdot
0 1 .872 r
.8424 .44518 Mdot
0 .633 1 r
.46015 .96437 Mdot
0 1 .873 r
.77379 .50818 Mdot
0 1 .881 r
.87754 .71742 Mdot
0 1 .87 r
.27708 .96397 Mdot
0 .939 1 r
.27707 .96408 Mdot
0 1 .861 r
.19212 .6996 Mdot
.589 0 1 r
.79815 .97169 Mdot
0 1 .869 r
.75224 .44323 Mdot
.27731 .87859 Mdot
0 1 .879 r
.73599 .71567 Mdot
0 1 .877 r
.67901 .69838 Mdot
0 1 .88 r
.7879 .87824 Mdot
0 1 .864 r
.63329 .39583 Mdot
0 1 .881 r
.82929 .71499 Mdot
.779 1 0 r
.46004 .97035 Mdot
0 1 .865 r
.62754 .44185 Mdot
0 1 .862 r
.81416 .96173 Mdot
0 1 .231 r
.27664 .97007 Mdot
0 1 .353 r
.81417 .96185 Mdot
0 1 .879 r
.76655 .87706 Mdot
0 1 .866 r
.3765 .71327 Mdot
0 1 .867 r
.72553 .39354 Mdot
0 1 .869 r
.28656 .87603 Mdot
0 1 .88 r
.8371 .71281 Mdot
0 1 .853 r
.1341 .43958 Mdot
0 1 .863 r
.59365 .43851 Mdot
.568 0 1 r
.81466 .96784 Mdot
0 1 .851 r
.11791 .3918 Mdot
0 1 .859 r
.08891 .39167 Mdot
0 1 .861 r
.19091 .71129 Mdot
0 1 .19 r
.75849 .19121 Mdot
0 1 .881 r
.91934 .69379 Mdot
0 1 .866 r
.61315 .50087 Mdot
0 1 .867 r
.7332 .3909 Mdot
0 1 .87 r
.29834 .95846 Mdot
0 .932 1 r
.29833 .95857 Mdot
0 1 .869 r
.82196 .39034 Mdot
0 1 .873 r
.44256 .9267 Mdot
0 1 .861 r
.21303 .69295 Mdot
0 1 .878 r
.7489 .92654 Mdot
0 1 .865 r
.64356 .43639 Mdot
0 1 .877 r
.67956 .71007 Mdot
0 1 .88 r
.86627 .69262 Mdot
0 1 .852 r
.1431 .4362 Mdot
0 1 .861 r
.92391 .87308 Mdot
0 1 .879 r
.783 .87292 Mdot
0 1 .865 r
.36655 .69191 Mdot
0 1 .873 r
.80375 .49861 Mdot
0 1 .851 r
.12689 .38832 Mdot
0 1 .878 r
.7307 .87223 Mdot
0 1 .877 r
.70393 .69116 Mdot
0 1 .206 r
.29794 .96457 Mdot
0 1 .87 r
.30766 .95604 Mdot
0 .929 1 r
.30766 .95615 Mdot
0 1 .869 r
.78212 .43329 Mdot
0 1 .861 r
.22219 .69004 Mdot
0 1 .88 r
.92077 .7055 Mdot
0 1 .869 r
.25319 .92241 Mdot
0 1 .858 r
.10945 .38367 Mdot
0 1 .196 r
.30728 .96216 Mdot
0 1 .863 r
.61824 .43006 Mdot
0 1 .861 r
.21189 .70467 Mdot
0 1 .88 r
.86751 .70434 Mdot
0 1 .877 r
.74383 .92139 Mdot
0 1 .866 r
.63784 .49271 Mdot
0 1 .877 r
.72025 .68644 Mdot
0 1 .865 r
.36597 .70363 Mdot
.71719 .38188 Mdot
0 1 .851 r
.09322 .4284 Mdot
0 1 .862 r
.88945 .92054 Mdot
0 1 .878 r
.77425 .92036 Mdot
0 1 .877 r
.70458 .70288 Mdot
0 1 .86 r
.5698 .38067 Mdot
0 1 .867 r
.76307 .38065 Mdot
0 1 .871 r
.41731 .86669 Mdot
0 1 .867 r
.23282 .86667 Mdot
0 1 .858 r
.11844 .38016 Mdot
0 1 .861 r
.22108 .70176 Mdot
0 1 .878 r
.75598 .86579 Mdot
0 1 .865 r
.60409 .48987 Mdot
.7249 .37921 Mdot
0 1 .862 r
.8974 .91864 Mdot
0 1 .872 r
.442 .94962 Mdot
0 .738 1 r
.442 .94973 Mdot
0 1 .863 r
.63434 .42453 Mdot
0 1 .876 r
.75038 .94946 Mdot
0 .276 1 r
.75039 .94957 Mdot
0 1 .878 r
.81149 .68223 Mdot
0 1 .866 r
.65401 .48737 Mdot
0 1 .863 r
.35551 .68172 Mdot
0 1 .852 r
.10804 .48702 Mdot
0 1 .869 r
.27452 .91667 Mdot
0 1 .878 r
.79084 .91632 Mdot
0 1 .877 r
.72096 .69818 Mdot
0 1 .867 r
.72749 .42225 Mdot
0 1 .879 r
.90962 .68045 Mdot
0 1 .861 r
.96772 .86232 Mdot
0 1 .87 r
.74749 .48551 Mdot
0 1 .878 r
.81936 .67999 Mdot
.434 1 0 r
.44185 .95576 Mdot
1 .042 0 r
.75078 .95559 Mdot
0 1 .872 r
.84664 .48491 Mdot
0 1 .877 r
.76925 .91517 Mdot
0 1 .878 r
.77253 .86158 Mdot
0 1 .85 r
.11388 .42052 Mdot
0 1 .858 r
.0846 .4204 Mdot
0 1 .867 r
.73524 .41963 Mdot
.2541 .86068 Mdot
0 1 .869 r
.28386 .91416 Mdot
0 1 .877 r
.71934 .86041 Mdot
0 1 .869 r
.82488 .41907 Mdot
0 1 .87 r
.75527 .48298 Mdot
0 1 .86 r
.59443 .3719 Mdot
0 1 .868 r
.25136 .94537 Mdot
0 1 .936 r
.25135 .94549 Mdot
1 0 .512 r
.48965 .14453 Mdot
0 1 .865 r
.6289 .48161 Mdot
0 1 .878 r
.81254 .69399 Mdot
0 1 .875 r
.74528 .94436 Mdot
0 .377 1 r
.74529 .94448 Mdot
0 1 .879 r
.8652 .85885 Mdot
0 1 .85 r
.12294 .41706 Mdot
0 1 .863 r
.35489 .69348 Mdot
0 1 .864 r
.70841 .36962 Mdot
1 0 .472 r
.48961 .14696 Mdot
0 1 .867 r
.26342 .85806 Mdot
0 1 .863 r
.89187 .94352 Mdot
0 1 .391 r
.89189 .94364 Mdot
0 1 .865 r
.75491 .36883 Mdot
0 1 .866 r
.2213 .85793 Mdot
0 1 .876 r
.7759 .94335 Mdot
0 .29 1 r
.77591 .94346 Mdot
0 1 .862 r
.92838 .91125 Mdot
0 1 .643 r
.25088 .95152 Mdot
0 1 .851 r
.24983 .36824 Mdot
0 1 .879 r
.91103 .69222 Mdot
0 1 .877 r
.78589 .91109 Mdot
0 1 .852 r
.12882 .47936 Mdot
0 1 .878 r
.82044 .69176 Mdot
0 1 .879 r
.87317 .85685 Mdot
0 1 .876 r
.733 .91041 Mdot
0 1 .863 r
.59456 .4783 Mdot
0 1 .864 r
.71616 .36691 Mdot
1 .374 0 r
.74567 .95052 Mdot
0 1 .863 r
.89989 .94164 Mdot
0 1 .86 r
.61055 .36616 Mdot
0 1 .396 r
.8999 .94176 Mdot
0 1 .873 r
.6565 .67297 Mdot
0 1 .867 r
.80582 .36596 Mdot
.441 0 1 r
.89252 .94968 Mdot
1 .087 0 r
.77635 .9495 Mdot
0 1 .858 r
.10531 .41244 Mdot
0 1 .865 r
.64514 .4762 Mdot
0 1 .851 r
.13794 .47601 Mdot
0 1 .878 r
.84997 .67125 Mdot
0 1 .868 r
.27283 .9397 Mdot
0 1 .865 r
.71908 .41066 Mdot
0 1 .938 r
.27282 .93981 Mdot
0 1 .877 r
.74479 .85386 Mdot
0 1 .876 r
.79261 .93935 Mdot
.426 0 1 r
.90055 .9478 Mdot
0 .3 1 r
.79262 .93946 Mdot
0 1 .878 r
.86053 .85338 Mdot
1 0 .554 r
.51402 .1346 Mdot
0 1 .859 r
.57023 .40945 Mdot
0 1 .867 r
.76542 .40943 Mdot
0 1 .857 r
.11439 .40895 Mdot
0 1 .87 r
.78557 .47313 Mdot
.977 1 0 r
.48946 .15823 Mdot
0 1 .875 r
.77088 .93821 Mdot
0 .389 1 r
.77089 .93832 Mdot
0 1 .865 r
.72687 .40801 Mdot
0 1 .866 r
.24266 .85187 Mdot
1 0 .515 r
.51401 .13704 Mdot
0 1 .634 r
.27237 .94586 Mdot
0 1 .878 r
.86853 .85137 Mdot
0 1 .868 r
.28224 .93721 Mdot
0 1 .94 r
.28223 .93732 Mdot
1 .122 0 r
.79309 .94552 Mdot
0 1 .871 r
.41608 .90497 Mdot
0 1 .867 r
.2295 .90495 Mdot
0 1 .873 r
.65698 .68477 Mdot
0 1 .876 r
.75858 .90409 Mdot
0 1 .863 r
.61949 .46992 Mdot
1 .414 0 r
.77131 .94438 Mdot
0 1 .877 r
.76145 .84957 Mdot
0 1 .873 r
.68173 .66552 Mdot
0 1 .878 r
.85117 .68305 Mdot
0 1 .866 r
.25203 .84921 Mdot
0 1 .879 r
.90421 .84907 Mdot
0 1 .631 r
.2818 .94338 Mdot
1 0 .585 r
.52998 .1281 Mdot
0 1 .864 r
.74632 .35639 Mdot
0 1 .865 r
.20918 .84873 Mdot
0 1 .849 r
.08735 .46828 Mdot
0 1 .863 r
.93109 .93433 Mdot
0 1 .414 r
.9311 .93445 Mdot
0 1 .876 r
.79718 .66417 Mdot
0 1 .875 r
.78763 .93418 Mdot
0 .398 1 r
.78764 .93429 Mdot
1 0 .547 r
.52998 .13054 Mdot
0 1 .875 r
.7344 .93351 Mdot
0 .433 1 r
.7344 .93362 Mdot
0 1 .866 r
.79787 .35396 Mdot
0 1 .863 r
.97273 .90067 Mdot
0 1 .869 r
.3934 .84649 Mdot
0 1 .86 r
.59511 .40073 Mdot
0 1 .876 r
.80511 .66187 Mdot
.947 1 0 r
.51396 .14831 Mdot
0 1 .876 r
.77533 .89994 Mdot
.364 0 1 r
.93182 .94051 Mdot
0 1 .863 r
.63581 .46444 Mdot
1 .444 0 r
.7881 .94036 Mdot
0 1 .873 r
.69825 .66064 Mdot
0 1 .867 r
.251 .89906 Mdot
1 .56 0 r
.73477 .93969 Mdot
1 0 .046 r
.47298 .12209 Mdot
0 1 .864 r
.71023 .39846 Mdot
0 1 .875 r
.72154 .89879 Mdot
0 1 .873 r
.68231 .67734 Mdot
0 1 .865 r
.75719 .39768 Mdot
0 1 .866 r
.73022 .46218 Mdot
0 1 .85 r
.24707 .39709 Mdot
0 1 .878 r
.89966 .84352 Mdot
1 0 0 r
.47293 .12453 Mdot
0 1 .877 r
.89379 .65874 Mdot
.86906 .89726 Mdot
0 1 .876 r
.79819 .67599 Mdot
0 1 .848 r
.23057 .34816 Mdot
0 1 .864 r
.71806 .39577 Mdot
0 1 .865 r
.23063 .84258 Mdot
0 1 .849 r
.10827 .46046 Mdot
0 1 .857 r
.07859 .46034 Mdot
0 1 .867 r
.26043 .89648 Mdot
.21783 .89636 Mdot
0 1 .86 r
.6114 .39502 Mdot
0 1 .867 r
.80862 .39483 Mdot
0 1 .866 r
.73809 .45958 Mdot
0 1 .877 r
.85057 .84172 Mdot
0 1 .87 r
.41533 .92812 Mdot
0 1 .867 r
.22749 .9281 Mdot
0 .899 1 r
.41533 .92823 Mdot
0 1 .821 r
.22748 .92822 Mdot
.925 1 0 r
.53 .14182 Mdot
0 1 .869 r
.82894 .45903 Mdot
0 1 .86 r
.32804 .65636 Mdot
0 1 .876 r
.70086 .84116 Mdot
0 1 .877 r
.87713 .89529 Mdot
0 1 .876 r
.80615 .67371 Mdot
0 1 .875 r
.76015 .92725 Mdot
0 .444 1 r
.76016 .92736 Mdot
0 1 .865 r
.24004 .83989 Mdot
0 1 .848 r
.11744 .45704 Mdot
0 1 .873 r
.69889 .67248 Mdot
0 1 .877 r
.85861 .83967 Mdot
0 1 .099 r
.41513 .93432 Mdot
0 .978 1 r
.22695 .9343 Mdot
.617 0 1 r
.46591 .11258 Mdot
0 1 .876 r
.836 .65293 Mdot
0 1 .877 r
.89516 .67059 Mdot
1 .597 0 r
.76058 .93345 Mdot
1 0 .095 r
.49754 .11194 Mdot
0 1 .875 r
.74729 .89235 Mdot
0 1 .878 r
.94863 .83793 Mdot
0 1 .864 r
.7895 .34132 Mdot
1 .696 0 r
.4727 .13582 Mdot
0 1 .876 r
.86435 .89188 Mdot
.643 0 1 r
.46586 .11502 Mdot
0 1 .863 r
.97576 .92386 Mdot
0 1 .443 r
.97577 .92398 Mdot
1 0 .05 r
.49751 .11439 Mdot
0 1 .857 r
.22241 .33964 Mdot
0 1 .875 r
.77702 .92315 Mdot
0 .453 1 r
.77703 .92326 Mdot
0 1 .857 r
.09956 .45245 Mdot
0 1 .86 r
.3273 .66821 Mdot
0 1 .861 r
.68657 .33909 Mdot
0 1 .866 r
.23942 .89039 Mdot
0 1 .87 r
.63592 .64976 Mdot
0 1 .876 r
.87244 .8899 Mdot
0 1 .867 r
.24913 .92228 Mdot
0 1 .819 r
.24912 .92239 Mdot
0 1 .874 r
.72286 .922 Mdot
0 .499 1 r
.72287 .92212 Mdot
0 1 .865 r
.72172 .45069 Mdot
0 1 .864 r
.74854 .38531 Mdot
.269 0 1 r
.97657 .93008 Mdot
0 1 .875 r
.72657 .83443 Mdot
0 1 .859 r
.57084 .44949 Mdot
0 1 .867 r
.76869 .44947 Mdot
0 1 .861 r
.69442 .33629 Mdot
1 .625 0 r
.77747 .92936 Mdot
1 0 .133 r
.51361 .1053 Mdot
0 1 .875 r
.76414 .88814 Mdot
0 1 .857 r
.10876 .44899 Mdot
0 1 .876 r
.87139 .92049 Mdot
0 .362 1 r
.8714 .92061 Mdot
0 1 .866 r
.24889 .88777 Mdot
0 1 .877 r
.90854 .88764 Mdot
0 1 .876 r
.83716 .6648 Mdot
0 1 .865 r
.8006 .38289 Mdot
0 .972 1 r
.24863 .92849 Mdot
0 1 .865 r
.72962 .44806 Mdot
0 1 .866 r
.20555 .88731 Mdot
0 1 .867 r
.25862 .91972 Mdot
0 1 .818 r
.25861 .91984 Mdot
1 .777 0 r
.72322 .92822 Mdot
0 1 .867 r
.21573 .9196 Mdot
0 1 .766 r
.21572 .91971 Mdot
1 0 .088 r
.5136 .10775 Mdot
0 1 .877 r
.94422 .8323 Mdot
.589 0 1 r
.49055 .10234 Mdot
0 1 .877 r
.88995 .8317 Mdot
0 1 .876 r
.87952 .91855 Mdot
0 .37 1 r
.87953 .91866 Mdot
.363 1 0 r
.4656 .12632 Mdot
1 .326 0 r
.87201 .92671 Mdot
1 .73 0 r
.49739 .12569 Mdot
.615 0 1 r
.49051 .10479 Mdot
0 1 .869 r
.39188 .8851 Mdot
0 1 .87 r
.63634 .66164 Mdot
0 .97 1 r
.25814 .92595 Mdot
0 1 .875 r
.7434 .83002 Mdot
0 .796 1 r
.21517 .92582 Mdot
0 1 .876 r
.84001 .82936 Mdot
1 .35 0 r
.88016 .92477 Mdot
0 1 .87 r
.66143 .64208 Mdot
1 0 .775 r
.60524 .09744 Mdot
0 1 .847 r
.22758 .37712 Mdot
0 1 .874 r
.74879 .91563 Mdot
0 .509 1 r
.7488 .91575 Mdot
0 1 .875 r
.86666 .91517 Mdot
0 .453 1 r
.86667 .91529 Mdot
0 1 .867 r
.37119 .82772 Mdot
0 1 .876 r
.90395 .88219 Mdot
.568 0 1 r
.50668 .09564 Mdot
0 1 .876 r
.84811 .82728 Mdot
1 0 .739 r
.60531 .09989 Mdot
0 1 .859 r
.59606 .44084 Mdot
0 1 .875 r
.88023 .64013 Mdot
1 0 .797 r
.613 .09428 Mdot
0 1 .866 r
.22723 .88126 Mdot
1 .757 0 r
.51355 .11905 Mdot
0 1 .867 r
.23746 .91369 Mdot
0 1 .762 r
.23745 .91381 Mdot
.594 0 1 r
.50666 .09809 Mdot
0 1 .861 r
.72495 .32542 Mdot
0 1 .863 r
.17901 .82582 Mdot
1 .81 0 r
.7492 .92187 Mdot
0 1 .875 r
.87481 .91321 Mdot
0 .459 1 r
.87482 .91333 Mdot
0 1 .875 r
.8543 .88041 Mdot
0 1 .863 r
.71276 .43859 Mdot
1 .624 0 r
.86728 .92141 Mdot
1 0 .762 r
.61308 .09673 Mdot
0 1 .874 r
.70287 .87986 Mdot
0 1 .865 r
.76037 .43781 Mdot
0 1 .873 r
.77615 .63762 Mdot
0 1 .874 r
.685 .82465 Mdot
.344 1 0 r
.49036 .1161 Mdot
0 1 .849 r
.24322 .43724 Mdot
0 1 .87 r
.67813 .63705 Mdot
0 1 .874 r
.76577 .91146 Mdot
0 .517 1 r
.76578 .91158 Mdot
0 .783 1 r
.23693 .91994 Mdot
0 1 .866 r
.23674 .87861 Mdot
0 1 .867 r
.24699 .9111 Mdot
0 1 .76 r
.24698 .91122 Mdot
0 1 .875 r
.86245 .87839 Mdot
.91116 .91097 Mdot
0 1 .864 r
.79217 .37032 Mdot
0 1 .87 r
.66194 .65399 Mdot
0 .4 1 r
.91117 .91109 Mdot
0 1 .863 r
.7207 .43593 Mdot
1 .646 0 r
.87544 .91945 Mdot
0 1 .866 r
.20335 .91064 Mdot
0 1 .709 r
.20334 .91076 Mdot
0 1 .859 r
.61257 .43518 Mdot
0 1 .873 r
.78418 .63525 Mdot
0 1 .867 r
.81251 .43499 Mdot
0 1 .856 r
.21933 .36865 Mdot
0 1 .876 r
.95351 .87668 Mdot
0 1 .86 r
.6882 .3681 Mdot
0 1 .875 r
.88156 .65204 Mdot
1 .835 0 r
.76621 .91771 Mdot
.792 1 0 r
.60565 .1112 Mdot
0 .777 1 r
.24648 .91736 Mdot
1 .451 0 r
.91186 .91722 Mdot
0 1 .869 r
.39095 .90846 Mdot
0 1 .95 r
.39095 .90857 Mdot
0 .609 1 r
.20276 .91689 Mdot
0 1 .876 r
.93482 .82028 Mdot
.329 1 0 r
.50657 .1094 Mdot
0 1 .86 r
.69613 .36532 Mdot
0 1 .863 r
.20068 .81947 Mdot
0 1 .873 r
.77709 .64955 Mdot
0 1 .876 r
.87967 .81916 Mdot
.776 1 0 r
.61346 .10804 Mdot
0 1 .87 r
.6787 .64898 Mdot
0 1 .866 r
.36025 .81848 Mdot
1 0 .89 r
.64323 .08196 Mdot
0 1 .874 r
.72889 .87324 Mdot
0 1 .597 r
.39071 .91471 Mdot
0 1 .874 r
.71094 .81776 Mdot
.90655 .90557 Mdot
0 .486 1 r
.90656 .90569 Mdot
0 1 .873 r
.78516 .64718 Mdot
1 0 .857 r
.64334 .08441 Mdot
0 1 .863 r
.21018 .81669 Mdot
0 1 .866 r
.22517 .90466 Mdot
0 1 .703 r
.22516 .90478 Mdot
0 1 .866 r
.61681 .62819 Mdot
0 1 .875 r
.94907 .87114 Mdot
0 1 .874 r
.85656 .90382 Mdot
0 .503 1 r
.85657 .90393 Mdot
0 1 .875 r
.89417 .87055 Mdot
0 1 .873 r
.70408 .90327 Mdot
0 1 .864 r
.75162 .42554 Mdot
0 .616 1 r
.70409 .90339 Mdot
1 .734 0 r
.90725 .91184 Mdot
0 1 .873 r
.81547 .626 Mdot
0 1 .861 r
.76866 .30985 Mdot
0 .588 1 r
.22462 .91093 Mdot
0 1 .866 r
.23474 .90203 Mdot
0 1 .7 r
.23473 .90215 Mdot
0 1 .874 r
.86477 .90182 Mdot
.74593 .8689 Mdot
0 .51 1 r
.86478 .90194 Mdot
0 1 .874 r
.72793 .81325 Mdot
0 1 .865 r
.80441 .42315 Mdot
1 .792 0 r
.85717 .91009 Mdot
1 0 .356 r
.58945 .07398 Mdot
0 1 .875 r
.84365 .86826 Mdot
.837 1 0 r
.70441 .90954 Mdot
0 1 .875 r
.95646 .90013 Mdot
0 .447 1 r
.95648 .90025 Mdot
1 0 .314 r
.58951 .07643 Mdot
0 1 .861 r
.72698 .3545 Mdot
0 1 .867 r
.36939 .86664 Mdot
0 .58 1 r
.2342 .90831 Mdot
1 .812 0 r
.86539 .9081 Mdot
0 1 .866 r
.61716 .64015 Mdot
1 0 .383 r
.59728 .07074 Mdot
0 1 .875 r
.85186 .8662 Mdot
.71 1 0 r
.64385 .09574 Mdot
1 0 .341 r
.59734 .0732 Mdot
0 1 .866 r
.64257 .6203 Mdot
0 1 .864 r
.17496 .86477 Mdot
0 1 .875 r
.82281 .80924 Mdot
1 .605 0 r
.95725 .90641 Mdot
0 1 .872 r
.81657 .63797 Mdot
0 1 .865 r
.34873 .80874 Mdot
0 1 .846 r
.22343 .41742 Mdot
0 1 .873 r
.73029 .89672 Mdot
0 .624 1 r
.7303 .89684 Mdot
0 1 .873 r
.68685 .86362 Mdot
0 1 .875 r
.92485 .80754 Mdot
.83101 .80709 Mdot
0 1 .865 r
.60739 .61756 Mdot
.966 0 1 r
.68651 .06433 Mdot
.441 0 1 r
.58276 .06403 Mdot
0 1 .874 r
.952 .89464 Mdot
0 .527 1 r
.95202 .89476 Mdot
.81 1 0 r
.73067 .90301 Mdot
1 .914 0 r
.58979 .08776 Mdot
0 1 .874 r
.89672 .89406 Mdot
.998 0 1 r
.68667 .06679 Mdot
0 .534 1 r
.89673 .89418 Mdot
.465 0 1 r
.58281 .06648 Mdot
0 1 .866 r
.65944 .61514 Mdot
0 1 .853 r
.09025 .6148 Mdot
.426 0 1 r
.59061 .06077 Mdot
0 1 .866 r
.64302 .63229 Mdot
0 1 .873 r
.74745 .89243 Mdot
0 1 .875 r
.93959 .85931 Mdot
0 .63 1 r
.74746 .89255 Mdot
0 1 .864 r
.79588 .41067 Mdot
1 .871 0 r
.95279 .90094 Mdot
0 1 .87 r
.75691 .61334 Mdot
1 .932 0 r
.59766 .08454 Mdot
.45 0 1 r
.59067 .06322 Mdot
0 1 .874 r
.84586 .89179 Mdot
0 .562 1 r
.84587 .89191 Mdot
0 1 .864 r
.19687 .85852 Mdot
0 1 .872 r
.86028 .61276 Mdot
1 .894 0 r
.89741 .90036 Mdot
1 0 .492 r
.62775 .05816 Mdot
0 1 .874 r
.8838 .85822 Mdot
0 1 .855 r
.21503 .40902 Mdot
0 1 .866 r
.35831 .85754 Mdot
0 1 .86 r
.69047 .40847 Mdot
0 1 .865 r
.6077 .62956 Mdot
1 0 .453 r
.62784 .06062 Mdot
0 1 .867 r
.3683 .89019 Mdot
0 1 .811 r
.36829 .89031 Mdot
0 1 .869 r
.76504 .6109 Mdot
0 1 .873 r
.71311 .85684 Mdot
.789 1 0 r
.74786 .89873 Mdot
0 1 .874 r
.85412 .88976 Mdot
0 .568 1 r
.85413 .88988 Mdot
0 1 .871 r
.6617 .80039 Mdot
1 .986 0 r
.84645 .89809 Mdot
0 1 .865 r
.63328 .60957 Mdot
0 1 .864 r
.20648 .85578 Mdot
0 1 .861 r
.77116 .33901 Mdot
0 1 .86 r
.6985 .40571 Mdot
0 1 .866 r
.65995 .62714 Mdot
0 1 .865 r
.17252 .88834 Mdot
0 1 .576 r
.1725 .88846 Mdot
0 1 .853 r
.08858 .62681 Mdot
.609 1 0 r
.68737 .07813 Mdot
.24 1 0 r
.58306 .07783 Mdot
0 1 .874 r
.86291 .79874 Mdot
0 .946 1 r
.368 .8965 Mdot
.995 1 0 r
.85473 .89607 Mdot
0 1 .872 r
.68797 .8872 Mdot
0 1 .852 r
.11184 .6074 Mdot
0 .724 1 r
.68798 .88732 Mdot
0 1 .869 r
.75779 .62535 Mdot
0 1 .872 r
.86156 .62478 Mdot
0 1 .863 r
.59748 .60637 Mdot
.229 1 0 r
.59096 .07457 Mdot
0 .169 1 r
.17186 .89466 Mdot
0 1 .873 r
.7303 .8524 Mdot
.364 0 1 r
.62118 .04807 Mdot
0 1 .869 r
.76595 .62292 Mdot
.48 1 0 r
.68827 .89352 Mdot
0 1 .865 r
.65023 .60434 Mdot
0 1 .852 r
.12131 .60415 Mdot
.99 1 0 r
.62829 .07196 Mdot
.387 0 1 r
.62127 .05052 Mdot
0 1 .865 r
.63369 .62159 Mdot
0 1 .873 r
.94248 .88294 Mdot
0 .573 1 r
.9425 .88305 Mdot
0 1 .871 r
.68797 .79326 Mdot
0 1 .869 r
.79669 .60137 Mdot
0 1 .865 r
.19456 .88216 Mdot
0 1 .565 r
.19455 .88227 Mdot
0 1 .853 r
.11025 .61943 Mdot
0 1 .873 r
.8863 .88185 Mdot
.82631 .84845 Mdot
0 .59 1 r
.88632 .88197 Mdot
0 1 .873 r
.80806 .79197 Mdot
0 1 .865 r
.34664 .84796 Mdot
0 1 .866 r
.35713 .88118 Mdot
0 1 .745 r
.35712 .8813 Mdot
0 1 .863 r
.59775 .6184 Mdot
0 1 .86 r
.72981 .39498 Mdot
0 1 .872 r
.71441 .88049 Mdot
0 .731 1 r
.71442 .88061 Mdot
.98 1 0 r
.94326 .88927 Mdot
1 0 .661 r
.67138 .04014 Mdot
0 1 .874 r
.92955 .84678 Mdot
0 1 .851 r
.48782 .27739 Mdot
0 1 .873 r
.8346 .84634 Mdot
0 1 .863 r
.62349 .59827 Mdot
0 .132 1 r
.19395 .88849 Mdot
0 1 .873 r
.81633 .78978 Mdot
0 1 .865 r
.20423 .87944 Mdot
.6507 .61638 Mdot
0 1 .56 r
.20422 .87956 Mdot
.921 1 0 r
.88698 .88818 Mdot
0 1 .853 r
.11975 .6162 Mdot
1 0 .624 r
.67152 .0426 Mdot
0 .726 1 r
.35681 .88752 Mdot
0 1 .871 r
.70517 .7886 Mdot
0 1 .85 r
.06848 .59668 Mdot
.458 1 0 r
.71477 .88683 Mdot
.186 1 0 r
.62169 .06188 Mdot
0 1 .869 r
.79774 .61342 Mdot
0 .116 1 r
.20363 .88578 Mdot
0 1 .874 r
.90861 .78678 Mdot
0 1 .872 r
.73173 .8761 Mdot
0 .736 1 r
.73174 .87621 Mdot
0 1 .863 r
.64052 .59296 Mdot
0 1 .862 r
.32004 .7845 Mdot
0 1 .863 r
.62387 .61033 Mdot
.269 0 1 r
.66497 .02988 Mdot
.442 1 0 r
.73212 .88244 Mdot
0 1 .867 r
.73902 .59077 Mdot
0 1 .871 r
.6633 .83974 Mdot
0 1 .851 r
.51341 .26762 Mdot
.0667 .60875 Mdot
.872 1 0 r
.67217 .05396 Mdot
.29 0 1 r
.6651 .03234 Mdot
0 1 .872 r
.82842 .87218 Mdot
0 .668 1 r
.82843 .8723 Mdot
0 1 .85 r
.09021 .58911 Mdot
0 1 .858 r
.05925 .58899 Mdot
0 1 .873 r
.86691 .83812 Mdot
0 1 .866 r
.34537 .8717 Mdot
0 1 .676 r
.34536 .87182 Mdot
0 1 .873 r
.84853 .78122 Mdot
0 1 .867 r
.74724 .58826 Mdot
0 1 .869 r
.84203 .58772 Mdot
0 1 .873 r
.9324 .87053 Mdot
0 .625 1 r
.93241 .87065 Mdot
0 1 .872 r
.83678 .87009 Mdot
0 .672 1 r
.83679 .87021 Mdot
0 1 .863 r
.64097 .60505 Mdot
.666 1 0 r
.82899 .87854 Mdot
0 1 .861 r
.77464 .37961 Mdot
0 1 .85 r
.09974 .58579 Mdot
0 .501 1 r
.34503 .87806 Mdot
0 1 .851 r
.53017 .26122 Mdot
0 1 .869 r
.64037 .77819 Mdot
0 1 .867 r
.73985 .60286 Mdot
.807 1 0 r
.93316 .87689 Mdot
.651 1 0 r
.83736 .87646 Mdot
0 1 .85 r
.08852 .60121 Mdot
0 1 .858 r
.05743 .60109 Mdot
.118 1 0 r
.66572 .04371 Mdot
0 1 .871 r
.68989 .83272 Mdot
0 1 .867 r
.7481 .60036 Mdot
0 1 .849 r
.48741 .3067 Mdot
0 1 .858 r
.08104 .58135 Mdot
0 1 .869 r
.84326 .59982 Mdot
0 1 .872 r
.81142 .83145 Mdot
0 1 .848 r
.47029 .2553 Mdot
0 1 .865 r
.73021 .57965 Mdot
0 1 .85 r
.09808 .5979 Mdot
0 1 .87 r
.66427 .86356 Mdot
0 .891 1 r
.66428 .86368 Mdot
0 1 .859 r
.57277 .57849 Mdot
0 1 .867 r
.77923 .57846 Mdot
0 1 .872 r
.81979 .82929 Mdot
0 1 .857 r
.0906 .578 Mdot
0 1 .865 r
.73846 .5771 Mdot
0 1 .871 r
.70731 .82813 Mdot
0 1 .872 r
.86933 .86196 Mdot
0 .69 1 r
.86934 .86208 Mdot
0 1 .869 r
.66694 .77084 Mdot
0 1 .072 r
.66453 .86994 Mdot
0 1 .873 r
.91318 .82635 Mdot
0 1 .858 r
.0793 .59348 Mdot
0 1 .873 r
.89467 .76897 Mdot
.591 1 0 r
.86998 .86834 Mdot
0 1 .865 r
.73101 .59177 Mdot
0 1 .85 r
.51327 .29697 Mdot
0 1 .856 r
.46285 .24594 Mdot
0 1 .863 r
.31756 .8241 Mdot
0 1 .86 r
.57295 .59062 Mdot
0 1 .867 r
.78022 .5906 Mdot
0 1 .848 r
.49608 .24531 Mdot
0 1 .871 r
.78636 .76657 Mdot
0 1 .858 r
.08889 .59014 Mdot
0 1 .869 r
.68434 .76603 Mdot
0 1 .865 r
.7393 .58923 Mdot
0 1 .87 r
.69106 .85661 Mdot
0 .895 1 r
.69106 .85673 Mdot
0 1 .859 r
.5991 .57011 Mdot
0 1 .871 r
.79474 .7643 Mdot
.81346 .85536 Mdot
0 .765 1 r
.81347 .85548 Mdot
0 1 .872 r
.85241 .82087 Mdot
0 1 .864 r
.72092 .56793 Mdot
0 1 .85 r
.53021 .2906 Mdot
0 1 .085 r
.69137 .86302 Mdot
0 1 .865 r
.77062 .56717 Mdot
0 1 .85 r
.23082 .56661 Mdot
0 1 .871 r
.82189 .85322 Mdot
0 1 .848 r
.51297 .23877 Mdot
0 .769 1 r
.8219 .85334 Mdot
.345 1 0 r
.814 .86176 Mdot
0 1 .864 r
.72922 .56534 Mdot
0 1 .87 r
.7086 .85206 Mdot
0 .898 1 r
.70861 .85218 Mdot
0 1 .869 r
.64174 .81788 Mdot
0 1 .86 r
.61635 .56462 Mdot
0 1 .867 r
.82506 .56443 Mdot
0 1 .857 r
.48872 .23585 Mdot
0 1 .86 r
.59939 .58227 Mdot
.333 1 0 r
.82245 .85963 Mdot
0 1 .872 r
.91596 .8503 Mdot
0 .719 1 r
.91597 .85042 Mdot
0 1 .846 r
.46969 .28471 Mdot
0 1 .094 r
.70895 .85848 Mdot
0 1 .867 r
.62053 .75753 Mdot
0 1 .864 r
.72169 .58009 Mdot
0 1 .865 r
.77158 .57934 Mdot
0 1 .864 r
.31606 .84807 Mdot
0 1 .516 r
.31606 .84819 Mdot
0 1 .85 r
.22966 .57878 Mdot
0 1 .852 r
.60927 .23102 Mdot
.498 1 0 r
.9167 .85672 Mdot
0 1 .871 r
.82738 .75544 Mdot
0 1 .864 r
.73002 .57752 Mdot
0 1 .857 r
.50567 .22925 Mdot
0 1 .86 r
.61671 .5768 Mdot
0 1 .867 r
.82624 .57661 Mdot
0 1 .848 r
.48685 .34752 Mdot
0 1 .853 r
.61744 .2279 Mdot
.029 0 1 r
.31566 .8545 Mdot
0 1 .869 r
.66864 .81064 Mdot
0 1 .871 r
.85475 .84488 Mdot
0 .783 1 r
.85476 .845 Mdot
0 1 .864 r
.76155 .55527 Mdot
0 1 .856 r
.46217 .27538 Mdot
0 1 .872 r
.89913 .80881 Mdot
0 1 .847 r
.49575 .27476 Mdot
.285 1 0 r
.85538 .85131 Mdot
0 1 .867 r
.64738 .74998 Mdot
0 1 .865 r
.81669 .55295 Mdot
0 1 .869 r
.64257 .84191 Mdot
0 1 .953 r
.64258 .84204 Mdot
0 1 .87 r
.78951 .80644 Mdot
0 1 .869 r
.68626 .8059 Mdot
0 1 .866 r
.61075 .74735 Mdot
0 1 .587 r
.6428 .84836 Mdot
0 1 .87 r
.798 .8042 Mdot
0 1 .849 r
.51308 .33786 Mdot
0 1 .864 r
.76248 .56748 Mdot
0 1 .847 r
.51283 .26825 Mdot
0 1 .867 r
.66496 .74503 Mdot
0 1 .847 r
.21001 .5474 Mdot
0 1 .853 r
.64925 .21576 Mdot
0 1 .858 r
.07216 .74471 Mdot
0 1 .865 r
.81784 .56517 Mdot
0 1 .856 r
.48832 .26534 Mdot
0 1 .869 r
.76648 .74331 Mdot
0 1 .871 r
.87415 .74275 Mdot
0 1 .869 r
.66967 .83475 Mdot
0 1 .951 r
.66967 .83487 Mdot
0 1 .849 r
.53026 .33153 Mdot
0 1 .869 r
.77496 .74096 Mdot
0 1 .871 r
.90183 .83293 Mdot
0 .805 1 r
.90185 .83305 Mdot
0 1 .851 r
.61016 .26053 Mdot
0 1 .866 r
.63773 .73969 Mdot
0 1 .867 r
.62168 .79754 Mdot
0 1 .592 r
.66995 .84121 Mdot
0 1 .864 r
.80787 .54085 Mdot
0 1 .847 r
.20875 .55963 Mdot
0 1 .85 r
.59272 .20788 Mdot
0 1 .856 r
.50545 .25876 Mdot
.20118 .53925 Mdot
0 1 .87 r
.79142 .83059 Mdot
0 1 .851 r
.61842 .25742 Mdot
0 .915 1 r
.79143 .83071 Mdot
0 1 .858 r
.09457 .73761 Mdot
0 1 .86 r
.69777 .53871 Mdot
.211 1 0 r
.90256 .83939 Mdot
0 1 .87 r
.83106 .79548 Mdot
0 1 .869 r
.68742 .83006 Mdot
0 1 .95 r
.68742 .83018 Mdot
0 1 .865 r
.60044 .73662 Mdot
0 1 .845 r
.46886 .32568 Mdot
0 1 .85 r
.60095 .20469 Mdot
0 1 .87 r
.79998 .82837 Mdot
0 .918 1 r
.79999 .82849 Mdot
0 1 .86 r
.70618 .53604 Mdot
0 1 .152 r
.79194 .83706 Mdot
0 1 .866 r
.6554 .73467 Mdot
0 1 .858 r
.1044 .7345 Mdot
0 1 .595 r
.68773 .83653 Mdot
0 1 .864 r
.809 .55311 Mdot
0 1 .159 r
.80051 .83485 Mdot
0 1 .854 r
.69483 .19836 Mdot
0 1 .869 r
.808 .73183 Mdot
0 1 .856 r
.19988 .55151 Mdot
0 1 .858 r
.58569 .19806 Mdot
0 1 .867 r
.64886 .7901 Mdot
0 1 .86 r
.69846 .55097 Mdot
0 1 .855 r
.46123 .31642 Mdot
0 1 .858 r
.59395 .19484 Mdot
0 1 .866 r
.61179 .78751 Mdot
0 1 .86 r
.7069 .54831 Mdot
0 1 .865 r
.62756 .72885 Mdot
0 1 .845 r
.4953 .3158 Mdot
0 1 .852 r
.65058 .24533 Mdot
0 1 .867 r
.62238 .82177 Mdot
0 1 .809 r
.62238 .82189 Mdot
0 1 .851 r
.63303 .19226 Mdot
0 1 .856 r
.04927 .72733 Mdot
0 1 .867 r
.66667 .78522 Mdot
0 1 .86 r
.06657 .7849 Mdot
0 1 .87 r
.83329 .81973 Mdot
0 .926 1 r
.8333 .81985 Mdot
0 1 .861 r
.73893 .52561 Mdot
0 .94 1 r
.62257 .82827 Mdot
0 1 .869 r
.76944 .78353 Mdot
0 1 .87 r
.87844 .78298 Mdot
0 1 .865 r
.64533 .72376 Mdot
0 1 .846 r
.51263 .30932 Mdot
0 1 .187 r
.83389 .82623 Mdot
0 1 .848 r
.59344 .23748 Mdot
0 1 .869 r
.77803 .78122 Mdot
0 1 .867 r
.74798 .72166 Mdot
0 1 .855 r
.48776 .30644 Mdot
0 1 .866 r
.6391 .77996 Mdot
0 1 .867 r
.64976 .8144 Mdot
0 1 .81 r
.64977 .81452 Mdot
0 1 .849 r
.60176 .2343 Mdot
0 1 .856 r
.07183 .72006 Mdot
0 1 .858 r
.62615 .18229 Mdot
0 1 .861 r
.03956 .71995 Mdot
.73979 .53792 Mdot
0 1 .867 r
.75656 .71924 Mdot
0 1 .86 r
.08923 .77791 Mdot
0 1 .869 r
.85535 .71873 Mdot
0 1 .865 r
.60136 .77694 Mdot
0 1 .866 r
.61242 .81183 Mdot
0 1 .85 r
.6114 .30165 Mdot
0 1 .74 r
.61242 .81196 Mdot
0 .943 1 r
.65 .82091 Mdot
0 1 .856 r
.08173 .71688 Mdot
0 1 .855 r
.50514 .2999 Mdot
0 1 .866 r
.657 .77502 Mdot
0 1 .86 r
.09916 .77484 Mdot
0 1 .853 r
.69667 .22799 Mdot
0 1 .867 r
.6677 .80957 Mdot
0 1 .85 r
.61978 .29857 Mdot
0 1 .811 r
.66771 .80969 Mdot
0 1 .857 r
.58634 .2277 Mdot
0 1 .861 r
.06318 .80925 Mdot
0 1 .232 r
.06316 .80938 Mdot
0 .712 1 r
.61259 .81835 Mdot
0 1 .852 r
.679 .17445 Mdot
0 1 .869 r
.77124 .80789 Mdot
0 1 .944 r
.77125 .80801 Mdot
0 1 .869 r
.8115 .77221 Mdot
0 1 .87 r
.88104 .80735 Mdot
0 .939 1 r
.88105 .80747 Mdot
0 1 .857 r
.59469 .22449 Mdot
0 1 .861 r
.06218 .71262 Mdot
.78588 .51068 Mdot
0 .945 1 r
.66798 .8161 Mdot
.966 0 1 r
.06227 .81578 Mdot
0 1 .869 r
.77989 .8056 Mdot
0 1 .943 r
.7799 .80572 Mdot
0 1 .866 r
.73885 .71098 Mdot
0 1 .849 r
.6342 .22192 Mdot
0 1 .615 r
.77172 .81442 Mdot
0 1 .865 r
.62882 .76928 Mdot
0 1 .231 r
.88174 .81388 Mdot
0 1 .862 r
.57474 .70986 Mdot
0 1 .867 r
.78996 .70984 Mdot
0 1 .866 r
.63994 .80436 Mdot
0 1 .743 r
.63994 .80448 Mdot
0 1 .861 r
.0721 .7094 Mdot
0 1 .866 r
.74747 .70853 Mdot
0 1 .859 r
.04332 .76778 Mdot
0 1 .618 r
.78039 .81214 Mdot
0 1 .861 r
.08599 .80233 Mdot
0 1 .208 r
.08597 .80245 Mdot
0 .719 1 r
.64016 .8109 Mdot
0 1 .859 r
.67228 .16431 Mdot
0 1 .866 r
.60192 .80136 Mdot
0 1 .67 r
.60192 .80148 Mdot
0 1 .861 r
.78693 .52303 Mdot
0 1 .851 r
.65243 .28655 Mdot
0 1 .865 r
.64681 .76426 Mdot
1 0 .953 r
.08512 .80887 Mdot
0 1 .866 r
.65797 .79946 Mdot
0 1 .744 r
.65798 .79958 Mdot
0 1 .861 r
.09599 .79929 Mdot
0 1 .197 r
.09598 .79941 Mdot
0 .478 1 r
.60207 .80791 Mdot
0 1 .857 r
.62724 .21199 Mdot
0 1 .867 r
.75076 .76219 Mdot
0 1 .862 r
.60221 .70181 Mdot
0 1 .868 r
.81362 .79668 Mdot
0 1 .94 r
.81363 .7968 Mdot
0 .724 1 r
.65823 .80601 Mdot
1 0 .919 r
.09514 .80584 Mdot
0 1 .858 r
.06614 .76062 Mdot
0 1 .863 r
.03346 .76051 Mdot
0 1 .867 r
.75944 .75981 Mdot
0 1 .865 r
.72923 .69972 Mdot
0 1 .847 r
.59444 .27874 Mdot
0 1 .869 r
.85947 .75931 Mdot
0 1 .866 r
.78106 .69899 Mdot
0 1 .628 r
.81419 .80324 Mdot
0 1 .866 r
.62959 .79378 Mdot
0 1 .856 r
.21819 .69845 Mdot
0 1 .673 r
.62959 .7939 Mdot
0 1 .858 r
.07614 .75748 Mdot
0 1 .851 r
.68069 .20418 Mdot
0 1 .865 r
.7379 .69723 Mdot
0 1 .847 r
.60289 .27558 Mdot
0 1 .86 r
.03972 .79229 Mdot
0 1 .119 r
.0397 .79241 Mdot
0 1 .862 r
.6202 .69654 Mdot
0 1 .867 r
.83784 .69636 Mdot
0 .489 1 r
.62979 .80034 Mdot
1 0 .661 r
.03875 .79886 Mdot
0 1 .862 r
.05634 .75328 Mdot
0 1 .866 r
.64772 .7888 Mdot
0 1 .675 r
.64772 .78893 Mdot
0 1 .852 r
.69923 .26931 Mdot
0 1 .856 r
.58724 .26902 Mdot
0 1 .866 r
.74153 .75166 Mdot
0 1 .867 r
.75244 .78675 Mdot
0 1 .815 r
.75244 .78688 Mdot
0 1 .863 r
.57535 .75056 Mdot
0 1 .867 r
.79329 .75054 Mdot
0 .497 1 r
.64796 .79538 Mdot
0 1 .862 r
.06637 .75011 Mdot
0 1 .858 r
.6739 .19407 Mdot
0 1 .856 r
.59572 .26583 Mdot
0 1 .86 r
.06269 .78519 Mdot
0 1 .09 r
.06267 .78532 Mdot
0 1 .863 r
.02976 .78508 Mdot
0 1 .443 r
.02974 .78521 Mdot
0 1 .866 r
.75027 .74925 Mdot
0 1 .867 r
.76119 .7844 Mdot
0 1 .816 r
.7612 .78452 Mdot
0 1 .868 r
.86197 .78389 Mdot
0 .959 1 r
.75289 .79334 Mdot
0 1 .936 r
.86199 .78402 Mdot
0 1 .865 r
.77167 .68756 Mdot
0 1 .848 r
.63582 .26328 Mdot
1 0 .566 r
.06176 .79178 Mdot
.269 0 1 r
.02877 .79167 Mdot
0 1 .86 r
.07276 .78208 Mdot
0 1 .078 r
.07274 .78221 Mdot
0 .961 1 r
.76166 .79099 Mdot
0 1 .866 r
.8292 .68532 Mdot
0 1 .849 r
.48503 .47947 Mdot
0 1 .643 r
.86264 .79049 Mdot
1 0 .526 r
.07186 .78868 Mdot
0 1 .863 r
.60317 .74263 Mdot
.0528 .77792 Mdot
0 1 .427 r
.05278 .77805 Mdot
0 1 .866 r
.73181 .74056 Mdot
0 1 .854 r
.19633 .67998 Mdot
0 1 .867 r
.74316 .77632 Mdot
0 1 .753 r
.74316 .77644 Mdot
0 1 .857 r
.62877 .2534 Mdot
0 1 .866 r
.78429 .73985 Mdot
0 1 .858 r
.21428 .73932 Mdot
0 1 .864 r
.57572 .77523 Mdot
0 1 .867 r
.7953 .77521 Mdot
0 1 .503 r
.57572 .77535 Mdot
0 1 .818 r
.79531 .77533 Mdot
0 1 .85 r
.48486 .49191 Mdot
.323 0 1 r
.05185 .78453 Mdot
0 1 .863 r
.0629 .77478 Mdot
0 1 .42 r
.06288 .7749 Mdot
0 1 .866 r
.74059 .73811 Mdot
0 1 .867 r
.75196 .77393 Mdot
0 1 .754 r
.75197 .77405 Mdot
0 1 .863 r
.6214 .73743 Mdot
0 .754 1 r
.74359 .78293 Mdot
0 1 .85 r
.51247 .47006 Mdot
0 1 .867 r
.84181 .73725 Mdot
.07 0 1 r
.57582 .78184 Mdot
0 .967 1 r
.79584 .78182 Mdot
.345 0 1 r
.06197 .78139 Mdot
0 .758 1 r
.75241 .78054 Mdot
0 1 .865 r
.8201 .67368 Mdot
0 1 .849 r
.68304 .24563 Mdot
0 1 .86 r
.18705 .67214 Mdot
0 1 .863 r
.70523 .67162 Mdot
0 1 .85 r
.53045 .4639 Mdot
0 1 .864 r
.60375 .76737 Mdot
0 1 .509 r
.60376 .76749 Mdot
0 1 .85 r
.51241 .48254 Mdot
0 1 .863 r
.71402 .66905 Mdot
0 1 .866 r
.73337 .76532 Mdot
0 1 .69 r
.73338 .76544 Mdot
0 1 .866 r
.77481 .72857 Mdot
0 1 .867 r
.78625 .76461 Mdot
0 1 .759 r
.78626 .76473 Mdot
.051 0 1 r
.60391 .774 Mdot
0 1 .86 r
.2119 .76408 Mdot
0 1 .101 r
.21189 .76421 Mdot
0 1 .846 r
.46619 .4582 Mdot
0 1 .857 r
.67616 .23557 Mdot
0 1 .866 r
.74222 .76289 Mdot
0 1 .691 r
.74223 .76302 Mdot
0 1 .867 r
.83309 .72637 Mdot
0 .544 1 r
.73379 .77196 Mdot
0 1 .864 r
.62212 .76221 Mdot
0 1 .514 r
.62213 .76234 Mdot
0 1 .867 r
.84421 .76204 Mdot
0 1 .821 r
.84422 .76217 Mdot
0 1 .85 r
.53046 .47639 Mdot
0 .773 1 r
.78678 .77125 Mdot
1 0 .601 r
.21127 .77073 Mdot
0 .55 1 r
.74266 .76954 Mdot
.037 0 1 r
.62232 .76886 Mdot
0 .978 1 r
.84486 .76869 Mdot
0 1 .857 r
.19208 .7211 Mdot
0 1 .847 r
.46593 .47071 Mdot
0 1 .863 r
.74825 .65902 Mdot
0 1 .856 r
.45818 .44917 Mdot
0 1 .847 r
.49385 .44857 Mdot
0 1 .866 r
.77672 .75343 Mdot
0 1 .698 r
.77673 .75356 Mdot
0 1 .866 r
.82389 .71489 Mdot
0 1 .867 r
.83544 .75125 Mdot
0 1 .766 r
.83545 .75137 Mdot
0 .573 1 r
.77723 .7601 Mdot
0 1 .847 r
.51198 .44225 Mdot
0 1 .862 r
.18266 .71336 Mdot
0 1 .856 r
.45789 .4617 Mdot
0 1 .864 r
.70754 .71286 Mdot
0 1 .847 r
.49371 .4611 Mdot
0 .796 1 r
.83607 .75792 Mdot
0 1 .856 r
.48595 .43944 Mdot
0 1 .864 r
.71645 .71032 Mdot
0 1 .859 r
.18951 .74603 Mdot
.035 1 0 r
.1895 .74615 Mdot
0 1 .851 r
.61542 .43477 Mdot
0 1 .847 r
.51192 .4548 Mdot
0 1 .863 r
.79736 .64463 Mdot
1 0 .151 r
.18882 .75271 Mdot
0 1 .856 r
.50414 .43306 Mdot
0 1 .851 r
.6242 .43176 Mdot
0 1 .856 r
.48577 .45199 Mdot
0 1 .866 r
.82619 .73987 Mdot
0 1 .709 r
.8262 .74 Mdot
0 1 .863 r
.18 .73836 Mdot
0 1 .356 r
.17999 .73849 Mdot
0 1 .864 r
.70895 .73786 Mdot
0 1 .54 r
.70895 .73798 Mdot
0 1 .852 r
.6158 .44734 Mdot
0 1 .864 r
.75115 .70042 Mdot
0 .609 1 r
.82681 .74657 Mdot
0 1 .856 r
.50405 .44563 Mdot
0 1 .864 r
.71792 .73534 Mdot
0 1 .543 r
.71793 .73547 Mdot
.557 0 1 r
.17929 .74506 Mdot
0 1 .852 r
.62462 .44434 Mdot
0 .05 1 r
.70932 .74456 Mdot
0 .061 1 r
.71832 .74205 Mdot
0 1 .852 r
.65843 .42003 Mdot
0 1 .865 r
.7529 .72553 Mdot
0 1 .556 r
.75291 .72565 Mdot
0 1 .848 r
.5977 .41241 Mdot
0 1 .852 r
.659 .43264 Mdot
0 1 .864 r
.80092 .68622 Mdot
0 1 .849 r
.60656 .40932 Mdot
0 .103 1 r
.75337 .73226 Mdot
0 1 .849 r
.59801 .42504 Mdot
0 1 .853 r
.70754 .4032 Mdot
0 1 .857 r
.59017 .40291 Mdot
0 1 .849 r
.60691 .42196 Mdot
0 1 .857 r
.59907 .39979 Mdot
0 1 .856 r
.48317 .61453 Mdot
0 1 .849 r
.6411 .3973 Mdot
0 1 .865 r
.80309 .71145 Mdot
0 1 .576 r
.8031 .71158 Mdot
0 1 .853 r
.70832 .41585 Mdot
0 1 .857 r
.59045 .41556 Mdot
.59938 .41245 Mdot
0 .169 1 r
.80367 .71822 Mdot
0 1 .856 r
.51184 .60545 Mdot
0 1 .85 r
.6416 .40996 Mdot
0 1 .857 r
.63374 .38764 Mdot
0 1 .856 r
.53063 .5995 Mdot
0 1 .85 r
.69066 .38004 Mdot
0 1 .858 r
.63421 .40033 Mdot
0 1 .854 r
.46344 .59399 Mdot
0 1 .858 r
.48259 .65651 Mdot
0 1 .851 r
.69138 .39275 Mdot
0 1 .858 r
.68349 .3702 Mdot
0 1 .86 r
.45506 .58526 Mdot
0 1 .854 r
.49236 .58468 Mdot
0 1 .858 r
.51165 .64754 Mdot
.68418 .38293 Mdot
0 1 .86 r
.48224 .68198 Mdot
0 1 .074 r
.48224 .68211 Mdot
0 1 .854 r
.51132 .57857 Mdot
0 1 .858 r
.53069 .64166 Mdot
0 1 .86 r
.48408 .57585 Mdot
1 0 .512 r
.48215 .68882 Mdot
0 1 .856 r
.46259 .63622 Mdot
0 1 .857 r
.61954 .57134 Mdot
0 1 .86 r
.51153 .67309 Mdot
0 1 .087 r
.51153 .67322 Mdot
0 1 .86 r
.50311 .56968 Mdot
0 1 .857 r
.62874 .56842 Mdot
1 0 .554 r
.51149 .67994 Mdot
0 1 .86 r
.53073 .66726 Mdot
0 1 .096 r
.53073 .66739 Mdot
0 1 .861 r
.45408 .6276 Mdot
0 1 .857 r
.4919 .62703 Mdot
1 0 .585 r
.53074 .67412 Mdot
0 1 .858 r
.46207 .66186 Mdot
.067 1 0 r
.46207 .66199 Mdot
0 1 .857 r
.6646 .55707 Mdot
.51112 .621 Mdot
1 0 .046 r
.46193 .66874 Mdot
0 1 .861 r
.4835 .61831 Mdot
0 1 .855 r
.60104 .54969 Mdot
0 1 .862 r
.45349 .65331 Mdot
0 1 .338 r
.45349 .65343 Mdot
0 1 .859 r
.62082 .61385 Mdot
0 1 .858 r
.49162 .65274 Mdot
.052 1 0 r
.49162 .65287 Mdot
0 1 .855 r
.61033 .5467 Mdot
0 1 .862 r
.50279 .61221 Mdot
0 1 .859 r
.63015 .61097 Mdot
.617 0 1 r
.45333 .6602 Mdot
1 0 .095 r
.49154 .65963 Mdot
0 1 .858 r
.51099 .64675 Mdot
.041 1 0 r
.51099 .64688 Mdot
0 1 .858 r
.71607 .54076 Mdot
0 1 .861 r
.59318 .54048 Mdot
0 1 .862 r
.48315 .64408 Mdot
0 1 .346 r
.48315 .64421 Mdot
1 0 .133 r
.51096 .65366 Mdot
0 1 .861 r
.6025 .53746 Mdot
.589 0 1 r
.48306 .651 Mdot
0 1 .856 r
.64653 .53504 Mdot
0 1 .86 r
.6216 .63966 Mdot
0 1 .154 r
.62161 .63979 Mdot
0 1 .86 r
.66652 .59975 Mdot
0 1 .862 r
.5026 .63803 Mdot
0 1 .353 r
.5026 .63816 Mdot
0 1 .861 r
.63101 .6368 Mdot
0 1 .161 r
.63101 .63693 Mdot
1 0 .775 r
.62181 .64659 Mdot
.568 0 1 r
.50255 .64496 Mdot
1 0 .797 r
.63124 .64374 Mdot
0 1 .858 r
.60209 .59245 Mdot
0 1 .861 r
.63884 .52568 Mdot
0 1 .858 r
.6115 .5895 Mdot
0 1 .861 r
.66768 .62566 Mdot
0 1 .189 r
.66769 .62579 Mdot
0 1 .856 r
.6985 .51831 Mdot
0 1 .86 r
.71873 .58363 Mdot
0 1 .862 r
.59412 .58335 Mdot
1 0 .89 r
.668 .63262 Mdot
0 1 .862 r
.60357 .58037 Mdot
0 1 .859 r
.60272 .61843 Mdot
0 1 .027 r
.60272 .61856 Mdot
0 1 .858 r
.64822 .57798 Mdot
0 1 .861 r
.69103 .50876 Mdot
0 1 .859 r
.61221 .61549 Mdot
0 1 .035 r
.61222 .61563 Mdot
1 0 .356 r
.60289 .6254 Mdot
1 0 .383 r
.6124 .62247 Mdot
0 1 .861 r
.72035 .60967 Mdot
0 1 .232 r
.72036 .6098 Mdot
0 1 .863 r
.59469 .6094 Mdot
0 1 .391 r
.59469 .60953 Mdot
0 1 .862 r
.64043 .56872 Mdot
0 1 .863 r
.60422 .60643 Mdot
0 1 .396 r
.60422 .60656 Mdot
.966 0 1 r
.72078 .61666 Mdot
.441 0 1 r
.59484 .61639 Mdot
0 1 .86 r
.64924 .60406 Mdot
0 1 .068 r
.64925 .60419 Mdot
.426 0 1 r
.60439 .61343 Mdot
0 1 .859 r
.70095 .56143 Mdot
1 0 .492 r
.64952 .61106 Mdot
0 1 .863 r
.6414 .59487 Mdot
0 1 .414 r
.64141 .595 Mdot
0 1 .863 r
.69338 .55199 Mdot
.364 0 1 r
.64166 .60189 Mdot
0 1 .86 r
.70244 .58763 Mdot
0 1 .119 r
.70244 .58776 Mdot
1 0 .661 r
.70284 .59467 Mdot
0 1 .863 r
.69481 .57826 Mdot
0 1 .443 r
.69482 .57839 Mdot
.269 0 1 r
.6952 .58531 Mdot
0 g
.25 Mabswid
.67245 0 m
.92713 .41497 L
s
.92713 .41497 m
1 .94814 L
s
1 .94814 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.08846 .24555 m
0 .80374 L
s
0 .80374 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.67245 0 m
.08846 .24555 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[48]:=",
  ImageSize->{265.312, 287.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004900017b000`400?l00000o`00003o/03oool2
000005L0oooo002^0?ooo`80000000<0oooo000000000000EP3oool00:/0oooo0`0000030?ooo`80
0000EP3oool00:T0oooo0P0000060?ooo`030000003oool0000005D0oooo002W0?ooo`800000203o
ool00`000000oooo0000001E0?ooo`00Y03oool3000000X0oooo00@000000?ooo`3oool00000E03o
ool00:80oooo0P00000=0?ooo`050000003oool0oooo0?ooo`000000D`3oool009l0oooo0`00000<
0?ooo`040440o`3oool0oooo000000<0oooo00<000000?ooo`3oool0D@3oool009d0oooo0P00000?
0?ooo`0404X0o`3oool0oooo000000@0oooo00<000000?ooo`3oool0D03oool009/0oooo0P00000A
0?ooo`0504X0o`3oool0oooo0?ooo`000000103oool00`000000oooo0?ooo`1?0?ooo`00V03oool3
000001D0oooo00<0o`2/0?ooo`000000103oool00`000000oooo0?ooo`1?0?ooo`00UP3oool20000
01L0oooo00@08?l00?l0W03oool000001@3oool00`000000oooo0?ooo`1>0?ooo`00T`3oool30000
00d0oooo00<0FP3o0?ooo`3oool02@3oool0100Ho`00o`2T0?ooo`0000050?ooo`030000003oool0
oooo04h0oooo002A0?ooo`800000403oool00`1R0?l0oooo0?ooo`0<0?ooo`030000003oool0oooo
00@0oooo00<000000?ooo`3oool0C@3oool008l0oooo0P00000B0?ooo`030680o`3oool0oooo00X0
oooo00<0i_l00?ooo`0000001`3oool00`000000oooo0?ooo`1<0?ooo`00S03oool3000000`0oooo
00<0JP3o0?ooo`3oool01`3oool00`3o07<0oooo0?ooo`080?ooo`030=ko003oool0000000L0oooo
00<000000?ooo`3oool0C03oool008X0oooo0P00000=0?ooo`0306X0o`3oool0L`3o00L0oooo00<0
<Ol00?ooo`3o07<0303oool00`000000oooo0?l0o`060?ooo`030000003oool0oooo04/0oooo0027
0?ooo`<000003`3oool00`1k0?l0oooo07<0o`070?ooo`03037o003oool0oooo00`0oooo00<00000
0?ooo`3o0?l01`3oool00`000000oooo0?ooo`1:0?ooo`00Q@3oool200000180oooo00D0L`3o0?oo
o`3oool0oooo0?l0HP070?ooo`030?oo003oool0oooo00X0oooo00<000000?ooo`3o0?l01`3oool0
0`000000oooo0?ooo`1:0?ooo`00P`3oool2000001D0oooo00@0o`1J03Wo003oool0o`1J1`3oool0
0`3oo`00oooo0?ooo`0:0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0B@3oool0
0800oooo0`00000F0?ooo`0503Wo003o05X0>Ol00?ooo`3o0580503oool00`000000oooo09co0008
0?ooo`030000003oool0oooo04T0oooo001n0?ooo`8000006@3oool00`0io`00oooo0?l0DP0=0?oo
o`030?l0iP3oool0oooo00H0oooo00<000000?ooo`2Do`002@3oool00`000000oooo0?ooo`180?oo
o`00N`3oool3000001l0oooo00<0onh00?ooo`3oool02@3oool00`3o0=h0oooo0?ooo`060?ooo`03
0000003oool0oooo00X0oooo00<000000?ooo`3oool0A`3oool007T0oooo0P00000P0?ooo`030?oZ
003oool0onh000/0oooo00<0o`3N0?ooo`3oool01P3oool00`000000oooo0?ooo`0:0?ooo`030000
003oool0oooo04L0oooo001g0?ooo`8000008P3oool00`3ojP00oooo0?ooo`030?ooo`030?l0c@3o
ool0oooo00h0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`160?ooo`00M03oool3
000000h0oooo00<0W03o0?ooo`3oool05`3oool00`3o0<D0oooo0?l0a@070?ooo`030;Co003oool0
oooo00H0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`150?ooo`00LP3oool20000
00`0oooo00<0W03o0?ooo`3oool00P3oool00`2D0?l0oooo0?ooo`0G0?ooo`030?l0_@3oool0o`35
00L0oooo00<0]?l00?ooo`3oool01`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo
04D0oooo001_0?ooo`<000003P3oool00`2L0?l0oooo0?ooo`030?ooo`030?l0803oool0oooo01H0
oooo00<0o`2m0?ooo`3oool04@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo04@0
oooo001]0?ooo`8000004@3oool00`2L0?l0oooo0?ooo`020?ooo`0305;o003o01P0oooo01T0oooo
00<0aOl00?ooo`3oool03`3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0oooo04@0oooo
001[0?ooo`8000003@3oool00`2L0?l0oooo0?ooo`050?ooo`050?l0403oool0oooo05;o003o0100
603oool00`3=o`00oooo0<Go000A0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool0
@`3oool006P0oooo0`00000?0?ooo`030:@0o`3oool0oooo00<0oooo00<0F_l00?ooo`3o0100703o
ool00`3=o`00oooo0?ooo`0A0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0@P3o
ool006H0oooo0P00000B0?ooo`030:@0o`3oool0oooo00<0oooo00<0D_l00?ooo`3o00P00`3oool0
0`3o_P00oooo0?ooo`0Z0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0@P3oool0
06<0oooo0`00000F0?ooo`030?l0003oool0oooo00L0oooo00<0ol800?ooo`3oool0:P3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo0440oooo001Q0?ooo`8000005`3oool00`1Jo`00
oooo0?l000050?ooo`030?nj003oool0oooo00H0oooo00<0o`2D0?ooo`3oool09@3oool00`000000
oooo0?ooo`0@0?ooo`030000003oool0oooo0400oooo001N0?ooo`<000006@3oool00`1Ro`00oooo
0?ooo`050?ooo`030?nj003oool0oooo00H0oooo00<0o`2;0?ooo`3oool09@3oool00`000000oooo
0?ooo`0@0?ooo`030000003oool0oooo0400oooo001L0?ooo`8000007P3oool00`3o/P00oooo0?oo
o`080?ooo`050?l0R`3oool0oooo0?ooo`3o08/09`3oool00`000000oooo0?ooo`0A0?ooo`030000
003oool0oooo03l0oooo001J0?ooo`800000803oool00`3o/P00oooo0?ooo`080?ooo`030?l0P`3o
ool0oooo02T0oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`0o0?ooo`00E`3oool3
000002d0oooo00D0o`230?ooo`3oool0oooo0>ko000W0?ooo`030000003oool0oooo0180oooo00<0
00000?ooo`3oool0?P3oool005D0oooo0P00000Y0?ooo`030?l0N`3oool0oooo00P0oooo00<0k_l0
0?ooo`3oool09@3oool00`000000oooo0?ooo`0C0?ooo`030000003oool0oooo03d0oooo001B0?oo
o`<00000:`3oool00`3o07<0oooo0?ooo`040?ooo`030?Ko003oool0oooo02T0oooo00<000000?oo
o`3oool04`3oool00`000000oooo0?ooo`0m0?ooo`00D03oool2000003D0oooo00<0k_l00?ooo`3o
ool0:@3oool00`000000oooo0?ooo`0D0?ooo`030000003oool0oooo03`0oooo001>0?ooo`800000
<03oool00`3fo`00oooo0?ooo`0`0?ooo`030000003oool0oooo01D0oooo00<000000?ooo`3oool0
>`3oool004/0oooo0`00000b0?ooo`030?oo003oool0oooo0300oooo00<000000?ooo`3oool05@3o
ool00`000000oooo0?ooo`0k0?ooo`00B@3oool2000006D0oooo00D00?oE0?ooo`3oool0oooo0000
000G0?ooo`030000003oool0oooo03X0oooo00160?ooo`<00000I`3oool01@00omh0oooo0?ooo`3o
ool0000001L0oooo00<000000?ooo`3oool0>P3oool004@0oooo0P00001/0?ooo`03003oe@3oool0
000001P0oooo00<000000?ooo`3oool0>@3oool00480oooo0P00001^0?ooo`03003ogP3oool00000
0180oooo00<002So0?ooo`3oool0103oool00`000000oooo0?ooo`0h0?ooo`00?`3oool300000780
oooo00<000000?ooo`3oool0403oool00`00:?l0oooo0?ooo`040?ooo`030000003oool0oooo03P0
oooo000m0?ooo`800000I@3oool00`00omh0oooo0?ooo`0=0?ooo`030000003oool0oooo0100oooo
00<002Co0?ooo`3oool01@3oool00`000000oooo0?ooo`0g0?ooo`00>P3oool3000006L0oooo00<0
0?oE0?ooo`3oool03@3oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo03H0oooo000h
0?ooo`800000K03oool00`00omh0oooo0?ooo`0;0?ooo`030000003oool0oooo0100oooo00<00?la
0?ooo`3oool01P3oool00`000000oooo0?ooo`0f0?ooo`00=P3oool2000006@0oooo00<00?oE0?oo
o`3oool01`3oool00`00omD0oooo0?ooo`080?ooo`04003ogP3oool0oooo000000L0oooo00<001Co
0?ooo`3oool0203oool00`00oc40oooo0?ooo`070?ooo`030000003oool0oooo03D0oooo000c0?oo
o`<00000H`3oool01000omh0oooo0?ooo`00omhD0?ooo`06003ogP3oool0oooo0000003oool00?oN
1@3oool00`006?l0oooo0?ooo`0B0?ooo`030000003oool0oooo03D0oooo000a0?ooo`800000IP3o
ool00`00omh0oooo0?ooo`020?ooo`03003oe@3oool0oooo0180oooo00@00?oN0000003oool00?oE
6`3oool00`000000oooo0?ooo`0d0?ooo`00;P3oool3000000`0oooo00<0R`3o0?ooo`3oool0F`3o
ool01000omD0oooo0?ooo`00omhC0?ooo`03000<o`00omD0000000L0oooo00<00?lY0?ooo`3oool0
503oool00`000000oooo0?ooo`0c0?ooo`00;03oool2000000l0oooo00<0U03o0?ooo`3oool0F`3o
ool00`00omh0oooo0?ooo`060?ooo`03003oe@3oool0oooo00/0oooo00<000Co0?ooo`0000001`3o
ool00`00ob00oooo0?ooo`0=0?ooo`03002Eo`3oool0oooo00@0oooo00<000000?ooo`3oool0<`3o
ool002X0oooo0P00000A0?ooo`0309@0o`3oool0oooo06@0oooo00<00?oN0?ooo`3oool00`3oool0
0`00omh0oooo0?ooo`050?ooo`030008o`3oool0000001L0oooo00<009Wo0?ooo`3oool01@3oool0
0`000000oooo0?ooo`0b0?ooo`009`3oool3000001D0oooo00<0o`0H0?ooo`3oool0J03oool00`00
omD0oooo0?ooo`070?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool0<@3oool002D0
oooo0P00000F0?ooo`0305;o003oool0o`0H06H0oooo00<00?oN0?ooo`3oool02@3oool00`00ob00
oooo0000000G0?ooo`03003oP`3oool00<ko00H0oooo00<000000?ooo`3oool0<@3oool00280oooo
0`00000H0?ooo`0305;o003oool0oooo05`0oooo00<00?oE0?ooo`3oool01`3oool00`00omD0oooo
0?ooo`090?ooo`03003o603oool0000000`0oooo00<0097o0?ooo`3oool0203oool00`00og/0oooo
003:o`070?ooo`030000003oool0oooo0300oooo000P0?ooo`8000007@3oool00`3oaP00oooo0?oo
o`120?ooo`03003ogP3oool0oooo01<0oooo00<00?oN0?ooo`00omh00`3oool00`00omh0oooo003o
gP0B0?ooo`030000003oool00?oE00T0oooo00<008go0?ooo`3oool02P3oool00`00b_l0oooo0?oo
o`050?ooo`030000003oool0oooo0300oooo000N0?ooo`8000007`3oool00`3oaP00oooo0?ooo`12
0?ooo`03003oe@3oool0oooo01<0oooo00T00?oE0?ooo`3oool0oooo003oe@080?l00?oE0?ooo`00
omD04P3oool00`000000oooo003ogP0H0?ooo`03003jo`3oool0oooo00@0oooo00<000000?ooo`3o
ool0;`3oool001/0oooo0`00000V0?ooo`030?l0W03oool0oooo03T0oooo00<00?oN0?ooo`3oool0
0`3oool00`00omh0oooo0?ooo`0C0?ooo`04003ogP3oool00?oN00P0oa40oooo00D00?oN0?ooo`3o
ool0oooo000000020?ooo`03002=o`3oool008Wo00L0oooo00<00?mk0036o`3oool02P3oool00`00
oi`00?[o0?ooo`060?ooo`030000003oool0oooo02h0oooo000I0?ooo`800000:@3oool00`3o08/0
oooo0?ooo`0i0?ooo`03003oe@3oool0oooo00<0oooo00<00?oE0?ooo`3oool04P3oool01@0@0?l0
0?oE0?ooo`3oool0403o0140oooo00D00?oN0?ooo`3oool0oooo000000020?ooo`030025o`3oool0
08Wo00L0oooo00<00?mc0036o`00`_l02P3oool00`00oj@00?Ko0?ooo`060?ooo`030000003oool0
oooo02h0oooo000G0?ooo`800000:`3oool00`3o09@0oooo0?ooo`0b0?ooo`03003ogP3oool0oooo
00H0oooo00<00?oN0?ooo`3oool05P3oool00`0@0?l0oooo0?ooo`0=0?ooo`03003oe@3oool0oooo
00D0oooo00<00?oE0?ooo`0000000P3oool00`00ROl0oooo0?ooo`090?ooo`030036o`3oool0oooo
0180oooo00<000000?ooo`3oool0;@3oool001L0oooo0P00000:0?ooo`0305X0o`3oool0oooo05<0
oooo00<00?oN0?ooo`3oool01P3oool00`00omD0oooo0?ooo`0@0?ooo`030100o`3oool0oooo00<0
oooo00D0403o0?ooo`3oool0oooo003o200;0?ooo`03003ogP3oool0oooo00D0oooo00<00?oN0?oo
o`000000103oool00`00og<0oooo0032o`090?ooo`03003fo`3oool0oooo00P0oooo00<00?o50?oo
o`3oool01P3oool00`000000oooo0?ooo`0/0?ooo`005`3oool00`000000oooo000000070?ooo`03
0580o`3oool0FP3o0200oooo00<0k_l00?ooo`3oool0CP3oool00`0@0?l0oooo0?ooo`070?ooo`03
003o203oool0oooo00<0oooo00<00?oN0?ooo`3oool03@3oool020000000oooo0?ooo`00og<0oooo
003oN`3oool00<;o1`3oool00`00oi`0oooo003jo`0:0?ooo`03003oa@3oool0oooo00H0oooo00<0
00000?ooo`3oool0;03oool001L0oooo00@000000?ooo`3oool000001@3oool20580o`040?ooo`1R
0?l0oooo0?l0Pah0oooo00<0i_l00?ooo`3oool0=03oool00`00omh0oooo0?ooo`0G0?ooo`0301P0
o`3oool0oooo00<0oooo00<00?l80?ooo`3oool0103oool01000omD0oooo0?ooo`00omh?0?ooo`08
0000003oool0oooo003oL`3oool00;ko0?ooo`00__l70?ooo`03003oW03oool00?Ko01@0oooo00<0
00000?ooo`3oool0:`3oool001H0oooo00<000000?ooo`3oool00P3oool00`110?l0oooo0?ooo`02
0?ooo`0605X0o`3oool0oooo0?l0R`3oool0o`1kE@3oool00`00omD0oooo0?ooo`060?ooo`03003o
gP3oool0oooo00@0oooo00<00?oE0?ooo`3oool03@3oool00`00o`P0oooo0?ooo`020?ooo`03003o
gP3oool00?oN0180oooo00<000000?ooo`3oool01P3oool00`00m_l0oooo0?ooo`0C0?ooo`03003o
Y03oool0oooo00@0oooo00<000000?ooo`3oool0:`3oool001H0oooo00<000000?ooo`3oool00P3o
ool00`1:0?l000000?ooo`040?ooo`0402Wo003o08<0oooo0?l0Neh0oooo00<00?oE0?ooo`3oool0
103oool00`00omh0oooo0?ooo`070?ooo`03003o203oool0oooo00P0oooo00<00?oE0?ooo`3oool0
103oool00`00omh0oooo0?ooo`0;0?ooo`030000003oool0oooo0080oooo00D00?nD0?ooo`00oi@0
oooo003bo`070?ooo`03003o_@3oool0oooo00/0oooo00<00?nL0?ooo`3oool01@3oool00`000000
oooo0?ooo`0Z0?ooo`005P3oool00`000000oooo0?ooo`020?ooo`0304X0o`3oool000000080oooo
00@0:Ol00?ooo`0ao`00o`23G03oool00`00omh0oooo0?ooo`030?ooo`03003ogP3oool0oooo00`0
oooo00<00?l00?ooo`3oool00P3oool00`00omh0oooo003ogP080?ooo`03001mo`3oool00?oE00d0
oooo00<000000?ooo`3oool00P3oool01@00oi`0oooo003oW03oool00?Ko00L0oooo00<00?o50?oo
o`3oool02`3oool00`00oj@0oooo0?ooo`060?ooo`030000003oool0oooo02T0oooo000F0?ooo`03
0000003oool0oooo00@0oooo00@0o`2L0?l0Y03oool08?l00`3oool00`3fo`00oooo0?ooo`1I0?oo
o`03003oe@3oool0oooo00<0oooo00<00?oE0?ooo`3oool04@3oool00`00omD0oooo003oe@040?oo
o`05001mo`3oool0oooo0?ooo`00OOl03`3oool010000000oooo0?ooo`00omD30?ooo`03003bo`3o
ool0oooo01h0oooo00<000000?ooo`3oool0:@3oool001H0oooo00<000000?ooo`3oool00P3oool0
1@0Ho`00oooo0?l0W03o0:`000000080oooo00<0m_l00?ooo`3oo`00E03oool00`00omh0oooo0?oo
o`060?ooo`03003ogP3oool0oooo01T0oooo00@00?oN0?ooo`3oool007Wo4`3oool010000000oooo
0?ooo`00omh30?ooo`03003oa@3oool00?o500/0oooo00<00?nT0?ooo`3oool0203oool00`00n_l0
oooo0?ooo`060?ooo`030000003oool0oooo02P0oooo000F0?ooo`030000003oool0oooo0080oooo
00<08?l00?ooo`3oool00P3oool00`000000oooo0>ko00070?ooo`030?l0gP3oool0oooo04`0oooo
00<00?oN0?ooo`3oool0103oool00`00omD0oooo003oe@0H0?ooo`04001io`00omD0oooo003oe@80
oooo00L007Wo0?ooo`3oool0oooo003oJP3oool00;Ko00d0oooo00<000000?ooo`3oool0103oool0
0`00okd0oooo003o_@0;0?ooo`03003oW03oool0oooo00P0oooo00<00?[o0?ooo`3oool01`3oool0
0`000000oooo0?ooo`0W0?ooo`005@3oool00`000000oooo0?ooo`050?ooo`050=ko003oool0oooo
0?ooo`0000001P3oool00`3o0>h0oooo0?l0iP1E0?ooo`03003ogP3oool0oooo01P0oooo00<007Wo
003ogP3oool01@3oool01@00]_l0oooo003oJP3oool00;Ko00d0oooo00<000000?ooo`3oool04P3o
ool00`00oj@0oooo0?ooo`0B0?ooo`030000003oool0oooo02L0oooo000E0?ooo`030000003oool0
oooo00D0oooo00<0g_l00?ooo`3oool00P3oool00`000000oooo0?ooo`030?ooo`030?l0iP3oool0
oooo0500oooo00<00?oE0?ooo`3oool07@3oool00`00NOl0oooo0?ooo`030?ooo`03003oJP3oool0
0;Ko00L0oooo00<00?oN0?ooo`3oool01`3oool00`000000oooo0?ooo`080?ooo`03003oW03oool0
0?nL00L0oooo00<00?Ko0?ooo`3oool0303oool00`00ocT0oooo0?ooo`040?ooo`030000003oool0
oooo02H0oooo000E0?ooo`030000003oool0oooo00X0oooo00<0o`3o0?H0o`3oool01@3oool00`2d
o`00oooo0?ooo`1>0?ooo`03003oe@3oool0oooo00d0oooo00<00?oE0?ooo`3oool03`3oool01`00
]_l0oooo0?ooo`3oool00?mZ0?ooo`00]_l00`3oool01@00oi@0oooo003bo`3oool00?oE00T0oooo
00<000000?ooo`3oool0203oool00`00oi`0oooo003oW0070?ooo`03003fo`3oool0oooo00`0oooo
00<00?li0?ooo`3oool0103oool00`000000oooo0?ooo`0V0?ooo`005@3oool00`000000oooo0?oo
o`0:0?ooo`030?l0o`3f0?l000000080oooo00@0[?l00?ooo`3oool0[?l0H03oool00`00omh0oooo
0?ooo`0=0?ooo`03003oJP3oool00;Ko00L0oooo00D00?;o0?ooo`00oi@0oooo003bo`0;0?ooo`03
0000003oool0oooo00P0oooo00<00?nL0?ooo`3oool05P3oool00`00ocT0oooo0?ooo`050?ooo`03
0000003oool0oooo02D0oooo000E0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`2/o`00
GP3oool00`00omh0oooo0?ooo`0B0?ooo`03003oHP3oool00;;o00D0oooo00<00?oE0?ooo`00k_l0
403oool00`000000oooo0?ooo`0:0?ooo`03003bo`3oool0oooo01`0oooo00<000000?ooo`3oool0
903oool001D0oooo00<000000?ooo`3oool02P3oool01@2Do`00oooo0?ooo`3oool0000005d0oooo
00<00?oE0?ooo`00omD06@3oool01P00omD0oooo0?ooo`00omh0oooo003bo`<0oooo00<00?nm0?oo
o`3oool02P3oool00`000000oooo0?ooo`080?ooo`03003fo`3oool00?Ko00X0oooo00<00?lY0?oo
o`3oool0203oool00`00[_l0oooo0?ooo`060?ooo`030000003oool0oooo02@0oooo000E0?ooo`03
0000003oool0oooo00X0oooo00<0W?l00?ooo`3oool00P3oool00`000000oooo0?ooo`1F0?ooo`05
003ogP3oool0oooo0?ooo`00omh0603oool01000oi@0oooo003^o`00omh80?ooo`03003o_@3oool0
oooo00X0oooo00<000000?ooo`3oool0203oool00`00l_l0oooo0?ooo`0:0?ooo`03003o<@3oool0
oooo00P0oooo00<00:ko0?ooo`3oool01`3oool00`000000oooo0?ooo`0S0?ooo`00503oool00`00
0000oooo0?ooo`0A0?ooo`030000003oool0oooo05D0oooo00<00?oE0?ooo`3oool00`3oool00`00
omD0oooo0?ooo`0D0?ooo`03003oU03oool00>ko00D0oooo00<00?nm0?ooo`3oool03P3oool00`00
0000oooo0?ooo`0A0?ooo`05003oe@3oool0oooo0?ooo`00obT05@3oool00`000000oooo0?ooo`0R
0?ooo`00503oool00`000000oooo0?ooo`0B0?ooo`030000003oool0oooo05X0oooo00<00?oN0?oo
o`3oool0703oool00`00okd0oooo0?ooo`060?ooo`03003oW03oool0oooo00D0oooo00<000000?oo
o`3oool0303oool00`00ob00oooo003o80020?ooo`03003ogP3oool0oooo0080oooo00<00:Go0?oo
o`3oool04P3oool00`000000oooo0?ooo`0R0?ooo`00503oool00`000000oooo0?ooo`0C0?ooo`03
0000003oool0oooo04`0oooo00<00?oN0?ooo`3oool01P3oool00`00omD0oooo0?ooo`0J0?ooo`03
003o_@3oool0oooo00P0oooo00D00?nD0?ooo`3oool0oooo003oU0070?ooo`030000003oool0oooo
00`0oooo00<00?lP0?ooo`00obT01`3oool00`00YOl0oooo0?ooo`0<0?ooo`03037o003oool0oooo
00@0oooo00<000000?ooo`3oool08@3oool001@0oooo00<000000?ooo`3oool0503oool00`000000
oooo0?ooo`1;0?ooo`03003oe@3oool0oooo00H0oooo00<00?oN0?ooo`3oool06P3oool00`00okd0
oooo0?ooo`080?ooo`05003oW03oool0oooo0?ooo`00oi`01`3oool00`000000oooo0?ooo`0<0?oo
o`03003o803oool0oooo01H0oooo00<0>Ol00?ooo`3oool0103oool00`000000oooo0?ooo`0Q0?oo
o`00503oool00`000000oooo0?ooo`0E0?ooo`030000003oool0oooo04`0oooo00<00?oE0?ooo`3o
ool0;03oool00`00oi@0oooo0?ooo`090?ooo`030000003oool0oooo00D0oooo00<00?oE0?ooo`3o
ool01P3oool00`00XOl0oooo0?ooo`0D0?ooo`0303Wo003oool0oooo00D0oooo00<000000?ooo`3o
ool0803oool001@0oooo00<000000?ooo`3oool05P3oool00`000000oooo0?ooo`1;0?ooo`03003o
e@3oool0oooo02H0oooo00<00?nL0?ooo`3oool01`3oool00`00l_l0oooo0?ooo`050?ooo`030000
003oool0oooo00D0oooo00<00?oN0?ooo`3oool0103oool00`00WOl0oooo002Qo`0F0?ooo`03001Q
o`3oool0oooo00H0oooo00<000000?ooo`3oool07`3oool001<0oooo00<000000?ooo`3oool05`3o
ool00`000000oooo0?ooo`1K0?ooo`03003ogP3oool0oooo01H0oooo00<00?nD0?ooo`3oool00`3o
ool01@00k_l0oooo0?ooo`3oool00>ko00L0oooo00<000000?ooo`3oool0303oool00`00XOl0oooo
0?ooo`0;0?ooo`0304[o003oool0oooo00P0oooo00<005go0?ooo`3oool01P3oool00`000000oooo
0?ooo`0O0?ooo`004`3oool00`000000oooo0?ooo`0H0?ooo`030000003oool0oooo02X0oooo00<0
203o0?ooo`3oool0;@3oool00`00omD0oooo0?ooo`0L0?ooo`03003bo`3oool0oooo00T0oooo00<0
00000?ooo`3oool06P3oool00`1:o`00oooo0?ooo`0;0?ooo`0307_o003oool0oooo00@0oooo00<0
00000?ooo`3oool07P3oool001<0oooo00<000000?ooo`3oool06@3oool00`000000oooo0?ooo`07
0?ooo`03003ogP3oool0oooo01l0oooo00<0203o0?ooo`3oool0:03oool00`00omD0oooo0?ooo`0K
0?ooo`03003^o`3oool0oooo00`0oooo00@00?lH003oe@3oool00000503oool00`00omh0D_l00?oo
o`0C0?ooo`0308?o003oool0oooo00D0oooo00<000000?ooo`3oool07@3oool001<0oooo00<00000
0?ooo`3oool06P3oool00`000000oooo0?ooo`060?ooo`03003ogP3oool0oooo04X0oooo00<00?oN
0?ooo`3oool06`3oool00`00l_l0oooo0?ooo`080?ooo`08003o403oool0oooo0?ooo`00oa000?oN
0?ooo`00000C0?ooo`0306;o0000omh0F_l00080oooo00<00?oN0?ooo`3oool00P3oool00`00EOl0
oooo0?ooo`0:0?ooo`0308?o003oool0oooo00D0oooo00<000000?ooo`3oool07@3oool001<0oooo
00<000000?ooo`3oool06`3oool00`000000oooo0?ooo`080?ooo`03003oe@3oool0oooo01`0oooo
00<00?l80?ooo`3oool08@3oool00`00omh0oooo0?ooo`0]0?ooo`03003o403oool0oooo00@0oooo
00<000000?ooo`3oool04@3oool00`1Jo`00oooo05;o00020?ooo`03003ogP3oool0oooo0080oooo
00<0053o0?ooo`3oool04`3oool00`000000oooo0?ooo`0L0?ooo`004`3oool00`000000oooo0?oo
o`0L0?ooo`030000003oool0oooo00L0oooo00<00?oN0?ooo`3oool0703oool00`00o`P0oooo0?oo
o`0Q0?ooo`03003oe@3oool0oooo02L0oooo00<00?l@0?ooo`3oool00`3oool01@00oa00oooo0?oo
o`3oool009Go0080oooo00<000000?ooo`3oool07P3oool00`2Lo`00oooo0?ooo`070?ooo`03000X
o`3oool0oooo00H0oooo00<000000?ooo`3oool0703oool00180oooo00<000000?ooo`3oool07P3o
ool00`000000oooo0?ooo`1c0?ooo`03003o403oool0oooo00L0oooo00@009Go0?ooo`3oool00000
203oool00`00omh0oooo0?ooo`080?ooo`030014o`3oool004So00D0oooo00<00?oN0?ooo`3oool0
0P3oool00`2Lo`00oooo0?ooo`070?ooo`03000/o`3oool0oooo00L0oooo00<000000?ooo`3oool0
6`3oool00180oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`1b0?ooo`03003o403o
ool0oooo00<0oooo00<0097o0?ooo`3oool0103oool00`000000oooo0?ooo`060?ooo`03003ogP3o
ool0oooo0080oooo00<00?oN0?ooo`3oool00`3oool00`00B?l0oooo0018o`020?ooo`040:Co003o
ool0oooo003ogQ80oooo00<0cOl00?ooo`3oool0103oool00`000000oooo0?ooo`0J0?ooo`004P3o
ool00`000000oooo0?ooo`0P0?ooo`030000003oool0oooo02T0oooo00<007go0?ooo`3oool0B`3o
ool00`00UOl0oooo0?ooo`050?ooo`030000003oool0oooo00X0oooo00<00?oN0?ooo`3oool01`3o
ool20:Co00030:co003oool0oooo00@0oooo00<001co0?ooo`3oool02`3oool00`3Eo`00oooo0?oo
o`040?ooo`030000003oool0oooo01X0oooo000B0?ooo`030000003oool0oooo0240oooo00<00000
0?ooo`3oool0:03oool00`00OOl0oooo0?ooo`0l0?ooo`03003ogP3oool0oooo00H0oooo00<0097o
0?ooo`3oool02`3oool00`000000Lol00?ooo`020?ooo`03003ogP3oool0oooo00l0oooo00<0]?l0
0?ooo`2/o`000`3oool01000omh0oooo0?ooo`006?l=0?ooo`030=Go003oool0oooo00D0oooo00<0
00000?ooo`3oool06@3oool00180oooo00<000000?ooo`3oool08P3oool00`000000oooo003ogP07
0?ooo`03003ogP3oool0oooo05`0oooo00<00?oE0?ooo`3oool01P3oool00`00TOl0oooo0?ooo`07
0?ooo`0:003ogP3oool00?oN0?ooo`000000Lol00?ooo`00omh0oooo003ogPT0oooo00<00?oN0?oo
o`3oool01@3oool00`2/o`00oooo003ogP030?ooo`03003ogP3oool0oooo01L0oooo00<000000?oo
o`3oool0603oool00180oooo00<000000?ooo`3oool08`3oool00`0000000?oN0?ooo`040?ooo`03
000Po`3oool00?oE01l0oooo00@00?mZ0?ooo`3oool00;[o=`3oool00`00omh0oooo0?ooo`0D0?oo
o`08003ogP1ko`000?oN0?ooo`000000Nol00?ooo`00omh;0?ooo`03003ogP3oool0oooo00D0oooo
00<000co0?ooo`00omh02P3oool00`3^o`00oooo0?ooo`070?ooo`0300P0o`3oool0m_l000H0oooo
00<000000?ooo`3oool0603oool00180oooo00<000000?ooo`3oool0903oool00`000000oooo003o
e@030?ooo`03000Lo`3oool001So01l0oooo00@00?mZ0?ooo`3oool00;Ko=`3oool00`00omD0oooo
0?ooo`0E0?ooo`0407?o003oool0oooo000001H0oooo00<000co0?ooo`003?l02P3oool00`3fo`00
oooo0?ooo`070?ooo`0300P0o`3oool0ool000L0oooo00<000000?ooo`3oool05`3oool00140oooo
00<000000?ooo`3oool02P3oool00`00omh0oooo0?ooo`0I0?ooo`03000/o`00omD0oooo0080oooo
00<001co0?ooo`3oool0903oool00`00l_l0oooo0?ooo`0/0?ooo`03003ogP3oool0oooo01H0oooo
00<0Pol00?ooo`3oool0103oool01000omD0oooo00000000=?l30?ooo`03003ogP3oool00?oN00T0
oooo00@00?oN0?ooo`3oool00?oN203oool00`3oo`00oooo0?ooo`040?ooo`030>ko003oool0oooo
00T0oooo00<0ool00?ooo`3oool01@3oool00`000000oooo0?ooo`0G0?ooo`004@3oool00`000000
oooo0?ooo`080?ooo`03003oe@3oool00?oE01/0oooo00<002Co0000003oool0103oool00`00obT0
oooo0?ooo`0P0?ooo`03003oU03oool00>ko02h0oooo00<00?oN0?ooo`3oool05P3oool00`1ko`00
oooo0?ooo`030?ooo`05003oe@00omh0oooo00000000=?l00`3oool00`00omh0oooo003ogP060?oo
o`04003ogP3oool0oooo003ogP80oooo00<00?oN0?ooo`3oool0103oool00`00omh0oooo0?oj0006
0?ooo`030200o`3oool0oooo0180oooo00<000000?ooo`3oool05P3oool00140oooo00<000000?oo
o`3oool0203oool01@00omh0oooo0?ooo`3oool00?oN01T0oooo00<002So0?ooo`0000000P3oool0
0`00obT0oooo003o:@0R0?ooo`03003oU03oool00?;o03l0oooo00<00?oN0?ooo`3oool02`3oool0
1000omh0oooo0?ooo`0000040?ooo`030=Go003oool0oooo00H0oooo00<00?oN0?ooo`3oool02P3o
ool00`00omh0oooo0?oj00060?ooo`030200o`3oool0omh000X0oooo00<0<@3o0?ooo`3ofP001P3o
ool00`000000oooo0?ooo`0E0?ooo`004@3oool00`000000oooo0?ooo`020?ooo`03003oe@3oool0
oooo00D0oooo00<00?oE0?ooo`00omD06@3oool01P00obT0oooo0?ooo`000000oooo003o:FH0oooo
00<00?oE0?ooo`3oool01@3oool00`00;?l0oooo0?ooo`040?ooo`04003ogP3oool000000=Go00<0
oooo00D0cOl00?ooo`00omh0oooo003ogP0A0?ooo`030?ob003oool0oooo00<0oooo00<00?oN0?oo
o`3oool00P3oool00`3ogP00oooo0?ooo`080?ooo`0302T0o`3oool0omH000H0oooo00<000000?oo
o`3oool05@3oool00140oooo00<000000?ooo`3oool00P3oool00`00omh0oooo0?ooo`050?ooo`03
003ogP3oool0oooo01T0oooo00L00?la0?ooo`3oool0oooo0000003oool00?oN00L0oooo00<009Go
0?ooo`3oool06`3oool00`00okd0oooo0?ooo`0g0?ooo`03003ogP3oool0oooo00<0oooo00<00?oN
0?ooo`00omh01@3oool00`00<?l0oooo0?ooo`040?ooo`04003oe@3oool000000=Go00<0oooo00D0
0?oN0?ooo`00omh0oooo003ogP060?ooo`03003ogP3oool0oooo00P0oooo00L0<@3o0?ooo`00omh0
oooo0?oB003oool00?oN0180oooo00<0oi400?ooo`3oool0103oool00`000000oooo0?ooo`0D0?oo
o`004@3oool00`000000oooo0?ooo`040?ooo`03003oe@3oool0oooo02@0oooo00<00000003oe@3o
ool0103oool00`00UOl0oooo002Ao`0M0?ooo`03003o_@3oool0oooo03<0oooo00L00?oE003ogP3o
ool0oooo003oe@3oool00?oN00<0oooo00<00?oE0?ooo`00omD02@3oool00`3Eo`00oooo0?ooo`02
0?ooo`060000003oool0oooo003ogP3oool00?oN0`3oool00`00omD0oooo0?ooo`040?ooo`03003o
gP3oool0oooo00P0oooo00D0<@3o0?ooo`00omh0oooo0?o600070?ooo`0303T0o`3o^P00oooo00T0
oooo00<0@@3o0?nA003oool01P3oool00`000000oooo0?ooo`0C0?ooo`00403oool00`000000oooo
0?ooo`050?ooo`03003ogP3oool0oooo02D0oooo00<000000?ooo`3oool00`3oool00`00UOl0oooo
002Ao`0R0?ooo`03003oU03oool0oooo02d0oooo0`00omh40?ooo`05003ogP3oool0oooo0?ooo`00
omh02`3oool00`3No`00oooo0?ooo`020?ooo`060000000@0?l0oooo003ogP3oool0203o403oool0
0`00omh0oooo0?ooo`040?ooo`030?o2003oool0olX000L0oooo00<0@@3o0?nj003o]P002@3oool0
0`1:0?l0oooo0?ooo`060?ooo`030000003oool0oooo01<0oooo000@0?ooo`030000003oool0oooo
00X0oooo00<00?oN0?ooo`3oool01@3oool00`00omh0oooo0?ooo`0@0?ooo`03003oe@3oool0oooo
00H0oooo00<009Wo0?ooo`3oool00P3oool00`00TOl0oooo0?ooo`020?ooo`030036o`3oool0oooo
01d0oooo00<00?nL0?ooo`3oool0;@3oool00`00omh0oooo0?ooo`0:0?ooo`03003ogP3oool0oooo
00X0oooo00H00?oN0?ooo`000000203o0?ooo`00omD40?ooo`03003ogP3odP00oooo00/0oooo00<0
0?oN0?ooo`3oool0403oool00`3o/P00ogD00?ooo`080?ooo`0307<0o`3oool0oooo00H0oooo00<0
00000?ooo`3oool04P3oool00100oooo00<000000?ooo`3oool0203oool00`00omh0oooo003ogP05
0?ooo`03003oe@3oool00?oE0180oooo00<00?oN0?ooo`3oool01P3oool00`00VOl000000?ooo`02
0?ooo`06003oN`3oool00?mk0036o`3oool00<Ko7`3oool00`00oi@0oooo0?ooo`0W0?ooo`03003o
gP3oool0oooo00<0oooo00D00?oN0?ooo`00omh0oooo003oe@080?ooo`03003ogP3oool0oooo00L0
oooo00H0403o0?ooo`3oool00?oN0?ooo`0000040?ooo`050?oB0000omh0oooo003ogP3ocP00203o
ool00`00omh0oooo0?ooo`050?ooo`04003ogP3oool0oooo0?ooo`80DP3o0P3oW@080?ooo`030?mY
003oool0oooo00L0oooo00<0L`3o0?ooo`3oool01P3oool00`000000oooo0?ooo`0B0?ooo`00403o
ool00`000000oooo0?ooo`080?ooo`03003ogP3oool0oooo00D0oooo00<00?oN0?ooo`3oool07@3o
ool02@000000oooo0?ooo`00og/0oooo003oL`00a_l0oooo0036o`190?ooo`03003ogP3oool0oooo
00<0oooo00D00?oN0?ooo`00omh0oooo003ogP040?ooo`05003ogP3oool0oooo0?ooo`00omD02@3o
ool00`0@0?l0oooo0?ooo`020?ooo`070000003oool0oooo003oe@3oool0olH0003ogP020?ooo`03
0?o>003oool0oooo0080oooo00D00?oN0?ooo`3oool0oooo003ogP070?ooo`09003ogP3oool0oooo
0?ooo`1B0?l0BP3o0?nE003oV@00oeT000H0oooo00<0FP3o0?mY003oool04@3oool2000001<0oooo
000@0?ooo`030000003oool0oooo00/0oooo00<00?oN003oe@3oool06@3oool00`00omh0oooo0?oo
o`040?ooo`04003oN`3oool0oooo003:o`H0oooo00<00?nL0?ooo`00m_l07@3oool00`00l_l0oooo
0?ooo`0]0?ooo`03003ogP3oool0oooo00H0oooo00<00?oN0?ooo`3oool0103oool00`00omh0oooo
003oe@080?ooo`070?o600000000oooo0?ooo`00omh0oooo0?o:00040?ooo`05003ogP3oool00?oN
0?ooo`00omh04@3oool0103oU@00oeD0003ogP3oD0020?ooo`05003ogP3oool0oooo0680o`2D0?l0
3`3oool3000000030?ooo`000000oooo0180oooo000@0?ooo`030000003oool0oooo0080oooo00<0
0?oN0?ooo`3oool01@3oool00`00omh00?oE003ogP0J0?ooo`03003oe@3oool0oooo00@0oooo00D0
0?n30?ooo`3oool00<[o000000020?ooo`06003oW03oool0oooo003oW03oool00?Ko7@3oool00`00
l_l0oooo0?ooo`0`0?ooo`800?oN2`3oool00`00omh0oooo003ogP080?ooo`030?nn00000000oooo
00H0oooo00D00?oN04X0o`00omh0oiD0003ogP0?0?ooo`03003ogP3oool0oooo0080oooo00<0odP0
003ogP3oC0000P3oool01@00omh0oooo0?ooo`3oool0U03o00`0oooo00<000000440o`000000103o
ool00`000000oooo0?ooo`0A0?ooo`00403oool00`000000oooo0?ooo`020?ooo`03003ogP3oool0
oooo00D0oooo00<00?oE0?ooo`3oool09P3oool00`00n_l0oooo003oY0020?ooo`03003fo`3oool0
oooo02<0oooo00<00?lH0?ooo`3oool0:@3oool00`00omD00?oN003oe@060?ooo`800?oN3P3oool0
0`3o_P00oooo000000030?ooo`070580o`3oool0oooo003ogP1B0?l0oooo0?nI00020?ooo`03003o
gP3oool0oooo00/0oooo00L00?oN0?ooo`3oool0oooo07<0o`3oB000JP3o0140oooo0`0000000`3o
ool0BP3o0?ooo`040?ooo`030000003oool0oooo0140oooo000?0?ooo`030000003oool0oooo00D0
oooo00<00?oN0?ooo`3oool0:@3oool01@00oj@0oooo0?ooo`00m_l0000000D0oooo00<00?o50?oo
o`3oool08@3oool00`00oaP0oooo0?ooo`0Y0?ooo`03003ogP3oool0oooo0080oooo00H00?oE0?oo
o`3oool0oooo003oe@00omh@0?ooo`050000003oool0oooo0?ooo`1:0?l00P3oool00`3oT@00oooo
0?ooo`030?ooo`03003ogP3oool0oooo00T0oooo00<00?oN0?ooo`3oool00`3oool0101c0?l0Y03o
0?ooo`2L0?l=0?ooo`<00000103oool00`1:0?l0oooo0?ooo`040?ooo`030000003oool0oooo0100
oooo000?0?ooo`030000003oool0oooo00D0oooo00<00?oE0?ooo`3oool0:@3oool01P00oi`0oooo
0?ooo`00n_l0oooo00000080oooo00<00?nm0?ooo`00okd08`3oool00`00oa00oooo0?ooo`0@0?oo
o`03003ogP3oool0oooo01T0oooo00D00?oN003oe@00omh0oooo003ogP080?ooo`03003oe@3oool0
oooo00H0oooo00@0FP3o0?ooo`000000ohd01@3oool01`3oS@00oooo0?ooo`3oool0L`3o0?ooo`3o
L@001`3oool01@00omh0oooo0?ooo`3oool00?oN00<0oooo00H00?oN0?ooo`3oool0[03o0?ooo`2T
0?l:0?ooo`0305X0o`000000000000P0oooo00<08?l00?ooo`3oool00`3oool00`000000oooo0?oo
o`0@0?ooo`003`3oool00`000000oooo0?ooo`0C0?ooo`03003ogP3oool0oooo01L0oooo00<00?oN
0?ooo`3oool01`3oool00`000000oooo003oa@070?ooo`03003oY03oool0oooo02h0oooo00<00?oE
0?ooo`3oool05@3oool01000omD0oooo0?ooo`3oool2003ogP80oooo00<00?oE0?ooo`3oool00P3o
ool01@00omh0oooo0?ooo`3oool00?oN00P0oooo0100DP3o0?ooo`000000ohD00?ooo`00omh0oooo
003ogP3oool0N`3o0?ooo`3oJ@00oooo07<0o`3oool0ofT00P3oool00`00omh0oooo0?ooo`020?oo
o`03003ogP3oool0oooo00D0oooo00<00?oN0?ooo`3oool02P3oool3000000030680o`3oool0oooo
00P0oooo00<06?l00?ooo`3oool00`3oool00`000000oooo0?ooo`0@0?ooo`003`3oool00`000000
oooo0?ooo`0@0?ooo`04003ogP3oool0oooo003oeAT0oooo00<00?oE0?ooo`3oool0103oool01@00
olD0oooo0?ooo`3oool0000000D0oooo00@00?nT0?ooo`3oool00?nL7@3oool00`00UOl0oooo0?oo
o`0;0?ooo`03003ogP3oool0oooo00@0oooo00<00?oN0?ooo`3oool04`3oool00`00omh0oooo0?oo
o`060?ooo`05003ogP3oool0oooo0?ooo`00omh01@3oool00`00omh0oooo0?ooo`060?ooo`0:0000
003oQ@00oooo003ogP3oool00?oN0?ooo`1c0?l0oooo0?mQ00<0oooo00H0ofT00?lD003oool00?oN
0?ooo`00onHC0?ooo`0407<0o`0000000000000000<0oooo00<0HP3o0?ooo`3oool03P3oool00`00
0000oooo0?ooo`0@0?ooo`003`3oool00`000000oooo0?ooo`0@0?ooo`03003oe@3oool0oooo0240
oooo00<00?nm0?ooo`3oool00P3oool00`000000oooo0?ooo`020?ooo`04003oW03oool0oooo003o
Y1d0oooo00<009Go0?ooo`3oool02`3oool00`00omh0oooo0?ooo`020?ooo`03003oe@3oool00?oE
01D0oooo00<00?oE0?ooo`3oool01@3oool00`00omD00?oN0?ooo`080?ooo`03003ogP3oool0oooo
00H0oooo00@0000007/0o`3oool0of401@3oool00`3oH@00oa@0003ogP020?ooo`05003ogP3oool0
0?oN0?ooo`00omh0403oool200000003003ogP1c0?l0oooo0080oooo00<00?oN0?ooo`3oool04@3o
ool00`000000oooo0?ooo`0@0?ooo`003`3oool00`000000oooo0?ooo`090?ooo`03003ogP3oool0
0?oE02h0oooo00<00?nT0?ooo`3oool0:@3oool00`00omh0oooo0?ooo`060?ooo`04003oe@3oool0
oooo003ogQ40oooo00<00?oN0?ooo`3oool01@3oool01000omh0oooo0?ooo`00omD20?ooo`03003o
gP3oool0oooo00@0oooo00<00?oN0?ooo`3oool02P3oool010000000N`3o0?ooo`3oE@040?ooo`07
003ogP3oool0o``0003ogP3oool0U03o003ogP030?ooo`03003ogP3oool0oooo00/0oooo0`000002
0?ooo`03003ogP1c0?l0oooo0080oooo00D00?oV0?ooo`3oool0oooo037o000?0?ooo`030000003o
ool0oooo0100oooo000>0?ooo`030000003oool0oooo00P0oooo00D00?oN0?ooo`00omh0oooo003o
gP0^0?ooo`07003oW0000000oooo0?ooo`00m_l0oooo003fo`0T0?ooo`0307?o0000omD0oooo00D0
oooo00<00?oN0?ooo`00omD0503oool00`00omD0oooo0?ooo`050?ooo`04003ogP3oool0oooo003o
gPD0oooo00D00?oN0?ooo`3oool0oooo003ogP0<0?ooo`060000003oool0oooo0?mE003oool00?oN
0P3oool00`00omh0W03o0?l800020?ooo`0309@0o`3=0?l0oooo0080oooo00<00?oN0?ooo`3oool0
203oool3000000D0oooo00<0N`3o0?ooo`3oool01P3oool00`0Yo`00oooo0?ooo`0=0?ooo`030000
003oool0oooo0100oooo000>0?ooo`030000003oool0oooo00P0oooo00<00?oN0?ooo`3oool00P3o
ool00`00omh0oooo0?ooo`0]0?ooo`050000003oool00?[o0?ooo`00n_l0903oool0101co`00oooo
0?ooo`00omh40?ooo`03003ogP3oool0oooo00@0oooo00<00?oN0?ooo`3oool0703oool00`00omh0
oooo003ogP0@0?ooo`030000003oool0oooo0080oooo00L00?oN0?ooo`3oool0oooo09@0o`3oool0
e@3o0080oooo00<0c@3o0?ooo`3oool02@3oool3000000X0oooo00<0>Ol00?ooo`3oool0503oool0
0`000000oooo0?ooo`0@0?ooo`003P3oool00`000000oooo0?ooo`020?ooo`03003ogP3oool0oooo
00D0oooo00@00?oN0?ooo`3oool00?oE;03oool01@00n_l0oooo0?ooo`3oool0000000P0oooo00<0
0?la0?ooo`3oool0703oool0101co`000?oE0?ooo`00omD60?ooo`03003oe@3oool0oooo0080oooo
00<00?oE0?ooo`3oool05`3oool00`00omh0oooo0?ooo`020?ooo`03003ogP3oool0oooo0100oooo
00H000000?ooo`3oool0oooo09`0o`3o00040?ooo`03003ogP3E0?l0oooo00X0oooo0`0000000`00
omh0oooo0?ooo`020?ooo`04003ogP3oool0oooo003ogP80oooo00<0>Ol00?ooo`0io`005`3oool0
0`000000oooo0?ooo`0?0?ooo`003P3oool00`000000oooo0?ooo`020?ooo`03003ogP3oool0oooo
00D0oooo00<00?oE0?ooo`3oool0;@3oool00`00n_l0oooo0?ooo`020?ooo`030000003oool0oooo
00<0oooo00@00?la0?ooo`00oc400?oN7@3oool00`00=?l00?oN0?ooo`070?ooo`03003oe@3oool0
oooo00L0oooo00<00?oE0?ooo`3oool04P3oool00`00omh0oooo0?ooo`040?ooo`03003ogP3oool0
oooo00h0oooo00H000000?ooo`3oool0oooo0:@0o`3N0?l20?ooo`03003ogP3oool00?oN00T0oooo
0`0000030?ooo`03003oiP3oool0oooo0080oooo00@00?oN0?ooo`3oool00?oN0P3oool00`0io`00
oooo0?ooo`050?ooo`03003ogP3oool0oooo00l0oooo00<000000?ooo`3oool03`3oool000h0oooo
00<000000?ooo`3oool01@3oool00`00omD0oooo0?ooo`0h0?ooo`03003o>@3oool0oooo0080oooo
00@00?la0?ooo`00obT00?oN7@3oool00`00>?l0oooo0?ooo`0A0?ooo`03003ogP3oool0oooo01T0
oooo00<00?oN0?ooo`3oool02P3oool01@00omh0oooo0?ooo`3oool0000000@0oooo00@0gP3o0?oo
o`3oool00?oN203oool00`0000000?oN09`0o`030?ooo`03003ogP3oool0oooo01<0oooo00<00?oN
0?ooo`3oool03`3oool00`000000oooo0?ooo`0?0?ooo`003P3oool00`000000oooo0?ooo`050?oo
o`03003ogP3oool0oooo03P0oooo00<00?li0000003oool0103oool00`00Z_l0oooo0?ooo`0O0?oo
o`03003ogP3=o`00oooo00T0oooo00<00?oN0?ooo`3oool06@3oool00`00omh0oooo0?ooo`0;0?oo
o`05003ogP3oool0oooo0?ooo`00onH00`3oool00`000000oooo0?ooo`0;0?ooo`070000002D0?l0
0?oV0?ooo`00onH0U03o0?ooo`03003ogRL0oooo00<000000?ooo`3oool03`3oool000h0oooo00<0
00000?ooo`3oool04P3oool00`00omh0oooo0?ooo`0[0?ooo`03003o>@3oool000000080oooo00<0
0:[o0?ooo`00YOl08@3oool00`00omD0cOl00?ooo`090?ooo`03003oe@3oool0oooo0080oooo00<0
0?oN0?ooo`3oool0503oool00`00omD0oooo0?ooo`040?ooo`04003ogP3oool0oooo003ogP<0oooo
00<00?oN0?ooo`3oool01@3oool00`000000oooo0?ooo`080?ooo`<0000000<0oooo09`0o`00omh0
103oool2003ogQ<0oooo00<00?oN0?ooo`3oool04P3oool00`000000oooo0?ooo`0?0?ooo`003@3o
ool00`000000oooo0?ooo`0A0?ooo`03003oe@3oool00?oE0300oooo00<000000?ooo`00Z_l0:03o
ool00`00omD0oooo0?ooo`090?ooo`03003oe@3oool0oooo00h0oooo00<00?oN0?ooo`3oool02P3o
ool01000omh00?oV0?ooo`00omh;0?ooo`030000003oool0oooo00D0oooo0P0000000`2T0?l0oooo
0?ooo`020?ooo`0509`0o`3oool0oooo0?ooo`1Bo`003P3oool00`00onH0oooo0?ooo`050?ooo`03
003oiP3oool0oooo0180oooo00<000000?ooo`3oool03`3oool000d0oooo00<000000?ooo`3oool0
4@3oool00`00omh0oooo0?ooo`0]0?ooo`05002^o`3oool0oooo0?ooo`000000203oool00`11o`00
oooo0?ooo`0J0?ooo`0500P0o`3oool0oooo0?ooo`00omh01P3oool00`00omD0oooo0?ooo`0C0?oo
o`03003oe@3oool0oooo00H0oooo00<00?oN0?ooo`3oool00P3oool00`00omh0oooo0?ooo`0;0?oo
o`030000003oool0oooo0080oooo0`0000020?ooo`070:@0o`3oool0oooo0?ooo`00omh0oooo05;o
00020?ooo`0305;o003oool0oooo00`0oooo00<00?oN0?ooo`3oool06P3oool00`000000oooo0?oo
o`0?0?ooo`003@3oool00`000000oooo0?ooo`0;0?ooo`03003oe@3oool0oooo03<0oooo00<00:ko
0?ooo`3oool00P3oool00`000000oooo0?ooo`030?ooo`03047o003oool0B_l001`0oooo00<0003o
0?ooo`3oool0203oool00`00omh0oooo0?ooo`0L0?ooo`03003ogP3oool0oooo00@0oooo00<00?oN
0?ooo`3oool02@3oool00`000000oooo00000002000000L0oooo00D00?oN0?ooo`00omh0oooo05[o
000:0?ooo`05003ogP3oool0oooo0?ooo`00omh01P3oool00`00onH0oooo0?ooo`080?ooo`03003o
gP3oool0oooo00`0oooo00<000000?ooo`3oool03P3oool000d0oooo00<000000?ooo`3oool02`3o
ool00`00omh0oooo0?ooo`0i0?ooo`0300000000omD0oooo0080oooo00@0@Ol00?ooo`11o`000?oN
7`3oool00`00omh0omH00?ooo`0L0?ooo`03003ogP3oool0oooo00/0oooo00<00?oN0?ooo`3oool0
203oool3000000P0oooo00<0F_l00?ooo`00omh0303oool00`00omh0oooo003ogP030?ooo`03003o
gP3oool0oooo00@0oooo00<00?oN0?ooo`3oool0203oool00`00onH0oooo0?ooo`0<0?ooo`030000
003oool0oooo00h0oooo000=0?ooo`030000003oool0oooo04P0oooo00D00?oN0?ooo`3oool0oooo
04[o00020?ooo`03003ogP3oool0oooo01d0oooo00<00?oE0?oB003oool0703oool00`00omh0oooo
0?ooo`060?ooo`03003ogP3oool0oooo00X0oooo00<00000003ogP0000000P3oool00`0000000?oV
0?ooo`060?ooo`0306;o003oool0oooo00`0oooo00<00?oN0?ooo`3oool04P3oool00`00onH0oooo
0?ooo`0?0?ooo`030000003oool0oooo00h0oooo000=0?ooo`030000003oool0oooo02h0oooo00<0
0?oE0?ooo`3oool05`3oool00`0io`00oooo0?ooo`030?ooo`03001@o`3oool0oooo01l0oooo00<0
olh00?ooo`3oool0903oool00`00omh0oooo0?ooo`060?ooo`@0000000<0oooo003ogP00omh00P3o
ool00`0000000?oN0?ooo`0B0?ooo`03003ogP3oool00?oN0080oooo00<00?oN0?ooo`3oool0403o
ool00`00omh0oooo0?ooo`0?0?ooo`030000003oool0oooo00h0oooo000<0?ooo`030000003oool0
oooo02d0oooo00<00?oN0?ooo`00omh06@3oool01`0io`0000000?ooo`3oool005Wo0?ooo`00EOl0
0`3oool00`00omh0U?l00?ooo`0K0?ooo`0404X0o`3oool0oooo0?nQ01d0oooo00<00?oN0?ooo`3o
ool02@3oool3000000030?ooo`00omh0oooo00<0oooo00@00?oV0?ooo`3oool00000503oool00`00
onH0oooo003oiP020?ooo`03003ogP3oool0oooo00P0oooo00<00?oV0?ooo`3oool02P3oool00`00
omh0oooo0?ooo`0:0?ooo`030000003oool0oooo00h0oooo000<0?ooo`030000003oool0oooo02d0
oooo00<00?oE0?ooo`3oool0603oool01P00HOl0oooo0?ooo`000000oooo001Eo`@0oooo00<0U?l0
003ogP2Lo`00703oool0101:0?l0oooo0?ooo`3oV@0M0?ooo`03003ogP3oool0oooo00H0oooo0`00
00030?ooo`800?oN1`3oool00`000000oooo0?ooo`0O0?ooo`04003ogP3oool0oooo003ogP`0oooo
00<00?oV0?ooo`3oool02P3oool00`000000oooo0?ooo`0>0?ooo`00303oool00`000000oooo0?oo
o`0X0?ooo`03003ogP3oool0oooo01d0oooo00H005go0?ooo`3oool0oooo00000023o`040?ooo`05
09Co003oool0U?l00?ooo`00omh07@3oool00`3oV@00oooo0?ooo`0Q0?ooo`<0000000<00?oV0?oo
o`3oool00`3oool00`00omh0oooo0?ooo`020?ooo`05003ogP3oool0oooo0?ooo`0000003`3oool0
0`00omh0oooo003ogP020?ooo`03003ogP3oool0oooo00H0oooo00D00?oN0?ooo`3oool0oooo003o
gP0L0?ooo`030000003oool0oooo00h0oooo000<0?ooo`030000003oool0oooo02P0oooo00<00?oN
0?ooo`3oool08P3oool00`23o`00oooo0?ooo`040?ooo`03000Lo`3oool00?oN01l0oooo00<0ogD0
0?ooo`3oool0703oool3000000<0oooo00<00?oN0?ooo`3oool00`3oool00`00omh0oooo0?ooo`02
0?ooo`05003ogP3oool0oooo0?ooo`000000103oool00`00omh0oooo0?ooo`080?ooo`03003ogP3o
ool00?oN0080oooo00<00?oE0?ooo`3oool0103oool00`00onH0oooo003oiP090?ooo`04003oiP3o
ool0oooo003ogQ@0oooo00<000000?ooo`3oool03@3oool000`0oooo00<000000?ooo`3oool0C@3o
ool01@23o`0000000?ooo`00omh002Co0080oooo00<001co0?ooo`3oool07@3oool00`1Z0?l0oooo
0?m]000K0?ooo`<00000203oool00`00omh0oooo0?ooo`080?ooo`04003ogP3oool00000003oiP<0
oooo00<00?oV0?ooo`3oool0403oool01@00omh0oooo003ogP3oool00?oN00/0oooo00@00?oN0?oo
o`3oool00?oV1@3oool01@00onH0oooo0?ooo`3oool00?oV00X0oooo00<000000?ooo`3oool03@3o
ool000`0oooo00<000000?ooo`3oool0C`3oool00`0000000?oN000Po`070?ooo`030>Ko003oool0
oooo01P0oooo00D0L`3o0?ooo`3oK@00oooo0?lH000F0?ooo`<000002`3oool00`00omh0oooo0?oo
o`080?ooo`04003ogP3oool00000003ogQH0oooo00<00?oN0?ooo`00omh05P3oool01P00omh0oooo
0?ooo`3oool00?oN003oiPT0oooo00<000000?ooo`3oool03@3oool000/0oooo00<000000?ooo`3o
ool0<03oool00`00omh0oooo0?ooo`040?ooo`03003ogP3oool0oooo01@0oooo00@002co0?ooo`3o
ool000001P3oool00`3Vo`00oooo0>ko000N0?ooo`030?lH003oool0oooo0140oooo0`0000080?oo
o`03003ogP3oool00?oN00/0oooo00<00?oV0?ooo`3oool00P3oool00`0000000?oV0?ooo`060?oo
o`03003ogP3oool0oooo00X0oooo00<00?oN0?ooo`3oool01@3oool00`00omh0oooo003ogP0E0?oo
o`03003ogP3oool0oooo00L0oooo00<000000?ooo`3oool03@3oool000/0oooo00<000000?ooo`3o
ool0;@3oool01000omh0oooo0?ooo`00omD40?ooo`03003ogP3oool00?oN0140oooo00<00?oN0?oo
o`3oool00P3oool01`00:?l0oooo0?ooo`3oool000000<go0000omh00`3oool00`3Vo`00oooo0>Ko
000L0?ooo`0309@0o`3oool0oooo0100oooo0`00000;0?ooo`03003ogP3oool00?oE00/0oooo00<0
0?oN0?ooo`00omh00P3oool00`000000oooo0?ooo`060?ooo`03003oe@3oool0oooo00X0oooo00<0
0?oN0?ooo`3oool01@3oool00`00onH00?oN003oiP050?ooo`05003oiP3oool0oooo0?ooo`00onH0
203oool00`00omh0oooo0?ooo`0:0?ooo`030000003oool0oooo00d0oooo000;0?ooo`030000003o
ool0oooo02d0oooo00@00?oN0?ooo`3oool00?oN103oool00`00omh0oooo0?ooo`0A0?ooo`03003o
gP3oool0oooo00L0oooo00<0eOl0003ogP3oool0103oool0100P0?l0oooo003ogP3ojP0I0?ooo`03
08/0o`3oool0a@3o00d0oooo0`00000N0?ooo`04003oiP3oool0oooo000001h0oooo00<00?oV0?oo
o`3oool0103oool01P00omh0oooo0?ooo`3oool00?oN003oiP80oooo00<00?oN0?ooo`3oool00P3o
ool00`00onH0oooo0?ooo`0:0?ooo`030000003oool0oooo00d0oooo000;0?ooo`030000003oool0
oooo02P0oooo00<00?oE0?ooo`3oool01@3oool00`00omh0oooo0?ooo`050?ooo`03003ogP3oool0
0?oN01P0oooo00D0eOl00000003oool0oooo01P0o`020?ooo`0401P0o`3olP000?oN0?oR01/0oooo
00<0c@3o0?ooo`3oool0203oool3000000T0oooo00<00?oE0?ooo`3oool02@3oool00`00omD0oooo
0?ooo`030?ooo`03003ogP3oool0oooo0080oooo00D00?oN0?ooo`3oool0oooo000000080?ooo`03
003ogP3oool0oooo00`0oooo00<00?oN0?ooo`3oool00`3oool00`00onH0oooo0?ooo`020?ooo`03
003ogP3oool0oooo00D0oooo00@00?oN0?ooo`3oool00?oN4@3oool00`000000oooo0?ooo`0=0?oo
o`002`3oool00`000000oooo0?ooo`0V0?ooo`03003ogP3oool00?oN00l0oooo00<00?oN0?ooo`00
omh06P3oool00`0000000?oN01P0o`030?ooo`030?oZ003oool0on8002<0oooo0`0000090?ooo`04
003ogP3oool0oooo003ogP/0oooo00<00?oN0?ooo`3oool00`3oool00`00onH00?oN0?ooo`020?oo
o`05003ogP3oool00?oN0?ooo`0000000P3oool00`00omh0oooo003oiP030?ooo`03003ogP3oool0
oooo00X0oooo00<00?oN0?ooo`00omh00P3oool01000omh0oooo003oiP00omh20?ooo`03003oiP3o
ool00?oV01`0oooo00<000000?ooo`3oool03@3oool000/0oooo00<000000?ooo`3oool09P3oool0
0`00omD0oooo0?ooo`0:0?ooo`03003ogP3oool0oooo00L0oooo00<00?oN0?ooo`3oool04`3oool0
10080?l0oooo0?ooo`00omh40?ooo`030?oZ003oool0oooo0080oooo00<0ol800?ooo`3oool06`3o
ool00`2;0?l000000000000<0?ooo`03003ogP3oool0oooo01<0oooo00<00?oN0?ooo`3oool00`3o
ool00`00omh0oooo000000020?ooo`03003oe@3oool00?oN00<0oooo00<00?oE0?ooo`3oool02P3o
ool00`00omh0oooo0?ooo`020?ooo`03003ogP3oool00?oN00<0oooo00@00?oN003oiP3oool00?oV
103oool00`00omh0oooo0?ooo`0:0?ooo`03003oiP3oool0oooo00P0oooo00<000000?ooo`3oool0
303oool000/0oooo00<000000?ooo`3oool0:03oool00`00omh0oooo0?ooo`080?ooo`03003ogP3o
ool0oooo00D0oooo00<00?oN0?ooo`00omh05@3oool01@080?l0oooo0?ooo`3oo`00000000<0oooo
00H0<@3o0?ooo`0i0?l0olh00?ooo`3o^P0J0?ooo`<0000000<0W03o0?ooo`3oool04@3oool00`00
omh0oooo0?ooo`080?ooo`03003ogP3oool0oooo00X0oooo00<00?oV00000000omh02P3oool00`00
omh0oooo0?ooo`0C0?ooo`03003ogP00onH00?oN00@0oooo00<00?oV0?ooo`3oool02P3oool00`00
omh0oooo0?ooo`020?ooo`03003ogP3oool0oooo00<0oooo00<000000?ooo`3oool0303oool000X0
oooo00<000000?ooo`3oool09`3oool00`00omD0oooo003oe@0>0?ooo`05003ogP3oool0oooo0?oo
o`00omh0703oool02P000000oooo0?ooo`0i0?l0oooo03T0o`3o`P00ohD00?nj003oL@0F0?ooo`<0
00000`3oool00`2D0?l0oooo0?ooo`0?0?ooo`03003oe@3oool00?oE00X0oooo00<00?oN0?ooo`3o
ool0203oool01@00omD00?oN003ogP0000000?oE00H0oooo00D00?oV0?ooo`3oool0oooo003oiP0D
0?ooo`03003ogP3oool00?oN01<0oooo00D00?oN0?ooo`3oool0oooo003oe@050?ooo`030000003o
ool0oooo00`0oooo000:0?ooo`030000003oool0oooo02L0oooo00<00?oN0?ooo`3oool02`3oool2
003ogP030?ooo`00omh0oooo01L0oooo00<00?oN0?ooo`3oool00`3oool0100a0?l0oooo0?ooo`3o
eP040?ooo`050?o6003oN@00oooo0?ma0000onH04@3oool4000001P0oooo00<00?oN0?ooo`3oool0
4`3oool01000onH0oooo003ogP00omh20?ooo`030000003oool0oooo00@0oooo00<00?oN0?ooo`3o
ool0103oool00`00omh0oooo0?ooo`090?ooo`07003oiP00omh0oooo003ogP00onH0oooo003oiP09
0?ooo`03003oiP3oool0oooo00@0oooo00<00?oN0?ooo`3oool00P3oool00`00onH0oooo0?ooo`07
0?ooo`030000003oool0oooo00`0oooo000:0?ooo`030000003oool0oooo02T0oooo00<00?oE0?oo
o`3oool02@3oool2003ogQX0oooo00<00?oE0?ooo`3oool00`3oool03@0Y0?l0oooo0?ooo`3oeP00
oiT00?ooo`00omh0DP3o0?ooo`3oO@00DP3o0?ooo`00omh03P3oool3000000X0oooo00<0D_l00?oo
o`3oool07`3oool00`00omD0oooo0?ooo`030?ooo`03003ogP3oool0oooo00<0oooo00<00000003o
iP3oool02`3oool00`00omh0oooo0?ooo`070?ooo`05003ogP3oool00?oN003ogP3oool00P00omh;
0?ooo`03003ogP3oool0oooo00@0oooo00<00?oV0?ooo`3oool0303oool00`000000oooo0?ooo`0<
0?ooo`002P3oool00`000000oooo0?ooo`0Y0?ooo`03003ogP3oool0oooo00@0oooo00<00?oN0?oo
o`3oool01`3oool00`00omh0oooo0?ooo`060?ooo`03003ogP3oool0oooo01D0oooo00@0oi400000
0000omh0DP3o0P3oool00`1J0?l0R`3o0?ooo`0;0?ooo`<000003@3oool00`1Bo`00oooo0?ooo`0O
0?ooo`03003ogP3oool0oooo00L0oooo00@00?oN0?ooo`0000000?oN1`3oool01@00omh0oooo0?oo
o`3oool00?oN00X0oooo00<00?oV0?ooo`3oool0403oool01@00omh0oooo0?ooo`3oool00?oN0100
oooo00<000000?ooo`3oool0303oool000X0oooo00<000000?ooo`3oool0;`3oool00`00omD00?oN
0?ooo`060?ooo`03003ogP3oool00?oN00H0oooo00<00?oN0?ooo`00omh0403oool00`00omh0oooo
0?ooo`030?ooo`040440o`3oT@00oooo00000080oooo00<0P`3o0?ooo`2;0?l02@3oool3000002H0
oooo00<00?oE0?ooo`3oool04P3oool02@00omh00?oE0?ooo`0000000?oN0?ooo`00omh0oooo003o
gP030?ooo`08003oiP3oool0oooo0?ooo`00onH00?oN003oiP00omh<0?ooo`07003ogP3oool00?oN
0?ooo`00onH0oooo003oiP020?ooo`03003ogP3oool0oooo0080oooo00D00?oV0?ooo`3oool0oooo
003ogP0@0?ooo`030000003oool0oooo00`0oooo000:0?ooo`030000003oool0oooo02d0oooo00@0
0?oN0?ooo`00omh00?oN0P3oool00`00omh0oooo0?ooo`020?ooo`03003ogP3oool0oooo0080oooo
00D00?oN0?ooo`3oool0oooo003ogP0B0?ooo`03003ogP3oool0oooo00<0oooo00L0@@3o07/0o`3o
ool0oooo0000003oool0P`3o00P0oooo0`00000;0?ooo`03003ogP3oool0oooo01H0oooo00<00?oN
0?ooo`3oool00P3oool01000omh0oooo0?ooo`00omh20?ooo`03003ogP3oool0oooo00`0oooo00@0
0?oN0?ooo`3oool000000P3oool00`00omh0oooo003ogP030?ooo`03003ogP3oool00?oN0080oooo
0`00omh<0?ooo`0<003ogP3oool00?oN0?ooo`00omh0oooo003ogP00onH0oooo003oiP3oool00?oN
2`3oool00`00onH0oooo0?ooo`0:0?ooo`030000003oool0oooo00/0oooo00090?ooo`030000003o
ool0oooo02T0oooo00<00?oN0?ooo`3oool00P3oool00`00omD0oooo0?ooo`030?ooo`03003ogP3o
ool0oooo00D0oooo0`00omhM0?ooo`0507<0o`3oool0oooo0?ooo`0000001P3oool20000000305X0
o`3oool0oooo00`0oooo00<00?oE0?ooo`3oool05P3oool00`00omh0oooo0?ooo`020?ooo`04003o
e@3oool0oooo003ogP80oooo00<00?oE0?ooo`3oool01P3oool00`00omh0oooo0?ooo`040?ooo`03
003ogP3oool000000080oooo00<00?oV0?ooo`3oool01@3oool00`00onH0oooo0?ooo`0D0?ooo`07
003ogP3oool00?oN0?ooo`00omD0oooo003oe@0;0?ooo`03003ogP3oool0oooo00X0oooo00<00000
0?ooo`3oool02`3oool000T0oooo00<000000?ooo`3oool02`3oool00`00omh0oooo0?ooo`0K0?oo
o`03003ogP3oool0oooo00<0oooo00<00?oE0?ooo`3oool02P3oool2003ogQX0oooo00<00?oN0?oo
o`3oool01P3oool00`000000oooo0?ooo`030000000405X0o`3oool0FP3o003oiPP0oooo00<00?oN
0?ooo`3oool0103oool00`00omh0oooo0?ooo`0=0?ooo`03003ogP3oool0oooo00T0oooo00<00?oN
0?ooo`3oool01@3oool00`00omh0oooo0?ooo`040?ooo`03003ogP3oool0oooo0080oooo00D00?mc
0?ooo`00onH0oooo000000020?ooo`06003ogP3oool00?oN003ogP3oool00?oN0P3oool00`00omh0
oooo0?ooo`050?ooo`03003ogP3oool0oooo00L0oooo00<00?oN003oiP00omh00P3oool00`00onH0
oooo0?ooo`0F0?ooo`03003ogP3oool0oooo00<0oooo00<000000?ooo`3oool02`3oool000T0oooo
00<000000?ooo`3oool0203oool01000omD0oooo0?ooo`00omhS0?ooo`03003ogP3oool0oooo00D0
oooo00<00?oN0?ooo`3oool07P3oool00`00omh0oooo0?ooo`070?ooo`030440o`00omh0oooo0080
oooo00@0FP3o0?ooo`1R0?l00?oN203oool00`00omh0oooo0?ooo`040?ooo`03003oe@3oool0oooo
00d0oooo00<00?oE0?ooo`3oool0103oool00`00omh00?oE0?ooo`020?ooo`03003oe@3oool0oooo
00D0oooo00<00?oE0?ooo`3oool02@3oool01@00ofX0oooo0?ooo`3oool000000080oooo00H00?oN
0?ooo`00onH00?oN003oiP00omh80?ooo`03003ogP3oool00?oN00T0oooo00<00?oV003ogP00omh0
2@3oool00`00onH0oooo0?ooo`0:0?ooo`03003ogP3oool0oooo0080oooo00<00?oN0?ooo`3oool0
0`3oool00`000000oooo0?ooo`0;0?ooo`002@3oool00`000000oooo0?ooo`080?ooo`03003ogP3o
ool0oooo0080oooo00<00?oE0?ooo`3oool09`3oool00`00omh0oooo0?ooo`0@0?ooo`03003oiP3o
ool0oooo01D0oooo00<0BP3o003ogP3oool0103oool00`0Yo`00oooo003ogP090?ooo`03003oe@3o
ool0oooo01P0oooo0`00omhG0?ooo`0304X0o`3oool0oooo0080000000@00?oN0?ooo`00omh0oooo
0`00omh90?ooo`03003ogP3oool0oooo01D0oooo00<00?oN0?ooo`3oool02P3oool00`00onH0oooo
0?ooo`080?ooo`030000003oool0oooo00/0oooo00090?ooo`030000003oool0oooo0080oooo00<0
0?oE0?ooo`3oool01@3oool01000omD00?oN0?ooo`00omhZ0?ooo`03003ogP3oool0oooo00d0oooo
00<00?oN0?ooo`00omh05`3oool00`1:0?l0oooo0?ooo`020?ooo`0502Wo003oool0:Ol00?ooo`00
omh02@3oool00`00omh0oooo0?ooo`0B0?ooo`03003oe@3oool0oooo00<0oooo00D00?oE0?ooo`3o
ool0oooo003oe@0E0?ooo`<000000P3oool00`000000oooo003ogP040?ooo`03003ogP3oool00?oN
00<0oooo00<00?oN0?ooo`3oool00P3oool00`00omh0oooo0?ooo`0;0?ooo`03003oiP3oool00?oN
00`0oooo00<00?oE0?ooo`3oool0103oool00`00onH0oooo0?ooo`070?ooo`030000003oool0oooo
00/0oooo00090?ooo`030000003oool0oooo0080oooo00<00?oN0?ooo`3oool00`3oool01000omh0
oooo003ogP00omDY0?ooo`04003ogP3oool0oooo003ogPP0oooo00@00?oN0?ooo`3oool00?oN0`3o
ool00`00onH0oooo0?ooo`0G0?ooo`03023o003oool0oooo0080oooo00<0:Ol00?ooo`3oool00`3o
ool00`00omD0oooo0?ooo`0J0?ooo`03003ogP3oool0oooo00D0oooo00<00?oN0?ooo`00omh01`3o
ool00`00omh0oooo0?ooo`070?ooo`@000000P3oool01@00ob00oooo0?ooo`00omh0000000<0oooo
00D00?oN0?ooo`00omh0oooo003ogP030?ooo`800?oV0`3oool2003ogPX0oooo00D00?oN0?ooo`00
omh0oooo003ogP070?ooo`03003ogP3oool0oooo0080oooo00<00?oN0?ooo`3oool0103oool00`00
omh0oooo0?ooo`070?ooo`030000003oool0oooo00/0oooo00090?ooo`030000003oool0oooo00D0
oooo00@00?oN0?ooo`3oool00?oE103oool00`00omh0oooo0?ooo`0U0?ooo`03003ogP3oool0oooo
00T0oooo00@00?oN0?ooo`00omh00?oN603oool00`00omh0oooo0?ooo`020?ooo`0301So003oool0
oooo00P0oooo0P00omhS0?ooo`03003oe@3oool0oooo00L0oooo00<00?mR0?ooo`3oool0103oool3
000000<0oooo00@00?oV0?ooo`3oool0o`2T0`3oool02@000000oooo0?ooo`3oool00?oN0?ooo`00
omh0oooo003ogP040?ooo`03003oiP00omD0oooo0080oooo00<00?oE0?ooo`3oool0203oool00`00
onH0oooo0?ooo`070?ooo`03003ogP3oool00?oV01D0oooo00<000000?ooo`3oool02`3oool000P0
oooo00<000000?ooo`3oool00`3oool01000omh0oooo0?ooo`00omD40?ooo`04003ogP3oool0oooo
003oeB80oooo00<00?oN0?ooo`3oool0203oool2003ogPH0oooo00<00?oN0?ooo`3oool05`3oool0
0`00omh0oooo0?ooo`020?ooo`030000003oool0oooo00T0oooo00<00?oV0?ooo`3oool03P3oool0
0`00omh0oooo0?ooo`0E0?ooo`03003ogP3oool0oooo0080oooo00@0HP3o0?ooo`3oool0oooo0`00
00000`00omh0oooo0?ooo`060?ooo`030?l0[03oool0oooo0080oooo00<00000003oiP3oool00`3o
ool2003ogP03003oiP3oool0oooo0080oooo00<00?oN0?ooo`3oool0103oool00`00omh0oooo0?oo
o`0:0?ooo`06003ogP3oool0oooo003ogP3oool00?oN0P3oool00`00onH0oooo0?ooo`0C0?ooo`03
0000003oool0oooo00X0oooo00080?ooo`030000003oool0oooo00<0oooo00<00?oE0?ooo`3oool0
1@3oool00`00omD0oooo0?ooo`060?ooo`03003ogP3oool0oooo01X0oooo00<00?oN0?ooo`3oool0
1@3oool00`00omh0oooo0?ooo`02003ogRD0oooo00<000000?ooo`3oool01@3oool00`00omh0oooo
0?ooo`0=0?ooo`03003oe@3oool0oooo0080oooo00<00?oN0?ooo`3oool05@3oool00`00omh0oooo
0?ooo`020?ooo`0405X0o`0000000000000000<0oooo00<00?oN0?ooo`3oool0303oool020000000
0?oV0?ooo`3oool00?oN003oiP00omh00?oV2`3oool00`00omh0oooo0?ooo`080?ooo`03003ogP3o
ool00?oN0080oooo00<00?oN0?ooo`3oool00P3oool00`00omh0oooo0?ooo`0C0?ooo`030000003o
ool0oooo00X0oooo00080?ooo`030000003oool0oooo00D0oooo00<00?oN0?ooo`3oool02@3oool0
1000omh0oooo0?ooo`00omDT0?ooo`03003ogP3oool0oooo01H0oooo00<00?oN0?ooo`3oool03P3o
ool00`000000oooo0?ooo`050?ooo`03003oiP3oool0oooo00d0oooo00<00?oN0?ooo`3oool04`3o
ool00`00ofX00?oN003oJP070?ooo`<0000000<0oooo003o4000omh04`3oool01P000000oooo0?oo
o`00omh0oooo003ogQL0oooo00@00?oV0?ooo`00onH00?oN703oool00`000000oooo0?ooo`0:0?oo
o`00203oool00`000000oooo0?ooo`050?ooo`03003oe@3oool0oooo00T0oooo00<00?oE0?ooo`3o
ool0803oool00`00omh0oooo0?ooo`050?ooo`03003ogP3oool00?oN00`0oooo00D00?oN0?ooo`3o
ool0oooo003ogP020?ooo`03003ogP3oool0oooo00h0oooo00<000000?ooo`3oool04`3oool00`00
omD0oooo0?ooo`0E0?ooo`03003oHP00omh00?mR00<0oooo100000001@00omh0oooo0?ooo`3oool0
o`1k00h0oooo00D00?oV0?ooo`3oool0oooo003o>@020?ooo`030000003oool0oooo00@0oooo00<0
0?oN0?ooo`00omh04`3oool00`00onH0oooo003oiP020?ooo`03003ogP3oool0oooo01L0oooo00<0
00000?ooo`3oool02P3oool000P0oooo00<000000?ooo`3oool0303oool00`00omD0oooo0?ooo`04
0?ooo`03003ogP3oool0oooo01h0oooo00<00?oN0?ooo`3oool01@3oool00`00omh0oooo003ogP09
0?ooo`04003ogP3oool0oooo003oiP<0oooo00<00?oV0?ooo`3oool04@3oool00`000000oooo0?oo
o`0=0?ooo`800?oN103oool00`00omh0oooo0?ooo`0E0?ooo`0307<0o`3oool0L`3o00<0000000<0
oooo003ogP00omh00P00omh30?ooo`040?l0P`00omh0oooo003ogP/0oooo00D00?oN0?ooo`3oool0
oooo0?H0o`020?ooo`030000003oool0oooo00<0oooo00@00?oV003ogP3oool00?oN203oool00`00
omh0oooo0?ooo`080?ooo`03003oiP3oool0oooo0080oooo00<00?oV0?ooo`3oool04@3oool00`00
omh0oooo0?ooo`030?ooo`030000003oool0oooo00X0oooo00080?ooo`030000003oool0oooo00`0
oooo00<00?oN0?ooo`3oool00P3oool00`00omD0oooo003oe@0S0?ooo`03003ogP3oool0oooo00h0
oooo00<00?oN0?ooo`00omh0603oool00`0000000?oN0?ooo`0=0?ooo`03003oe@00onH0oooo01/0
oooo0`0000001@00omh0oooo003o203oool00?oN00D0oooo00D00?oN0?ooo`00omh0oooo003ogP0?
0?ooo`060?H0o`3oool0oooo0?ooo`0000000?oN0`3oool00`00omh0oooo0?ooo`050?ooo`05003o
e@3oool0oooo0?ooo`00omh08`3oool00`00omD0oooo0?ooo`030?ooo`030000003oool0oooo00X0
oooo00070?ooo`030000003oool0oooo0180oooo00<00?oN0?ooo`3oool08`3oool00`00omh0oooo
0?ooo`090?ooo`03003oiP3oool0oooo00@0oooo00<00?oN0?ooo`3oool05P3oool00`0000000?oN
0?ooo`090?ooo`05003oiP3oool0oooo0?ooo`00omh01`3oool00`00omh0oooo0?ooo`0@0?ooo`<0
000000H0oooo003ogP00o`P00?oN0?ooo`3o06840?ooo`04003ogP3oool0oooo003ogQP0oooo00<0
00000?ooo`3oool0103oool2003ogP<0oooo00<00?oN0?ooo`3oool0:`3oool00`000000oooo0?oo
o`0:0?ooo`001`3oool00`000000oooo0?ooo`0<0?ooo`03003oe@3oool0oooo03D0oooo00<00?oN
0?ooo`3oool03P3oool00`00omh0oooo003ogP0<0?ooo`030000003oool0oooo00H0oooo00@00?oN
0?ooo`3oool00?oN1P3oool2003ogP<0oooo00<00?oE0?ooo`3oool03@3oool3000000D0oooo00D0
o`1J0?ooo`3oool0o`1R003ogP030?ooo`03003ogP3oool0oooo00@0oooo00<00?oN0?ooo`00oc40
303oool00`00omh0oooo0?ooo`040?ooo`080000003oool0oooo0?ooo`00omh0oooo003ogP00omD;
0?ooo`03003ogP3oool0oooo00T0oooo00<00?oN0?ooo`3oool01P3oool00`00omh0oooo0?ooo`0?
0?ooo`030000003oool0oooo00T0oooo00070?ooo`030000003oool0oooo00`0oooo00<00?oN0?oo
o`3oool0=P3oool00`00omh0oooo0?ooo`0=0?ooo`03003ogP3oool00?oN00`0oooo00<000000?oo
o`3oool01P3oool00`00omh0oooo0?ooo`070?ooo`03003ogP00onH0oooo00h0oooo100000080?oo
o`050?l0FP00omh0oooo0?ooo`00omh02P3oool00`00omh0oooo003o<@0<0?ooo`03003oiP3oool0
oooo00D0oooo00L00000003ogP3oool00?oN0?ooo`00omh00?oV00/0oooo00<00?oV0?ooo`3oool0
2@3oool00`00onH0oooo0?ooo`060?ooo`03003ogP3oool0oooo00l0oooo00<000000?ooo`3oool0
2@3oool000L0oooo00<000000?ooo`3oool0@`3oool00`00omh0oooo003ogP0:0?ooo`03003ogP3o
ool0oooo0100oooo00<000000?ooo`3oool07`3oool3000000d0oooo00<00?oN0?ooo`3oool01`3o
ool00`00omh0oooo0?ooo`030?ooo`030?l0iP3oool0oooo01<0oooo00<000000?ooo`3oool00P3o
ool01@00omh0oooo0?ooo`3oool00?oN01D0oooo00<00?oE0?ooo`3oool05P3oool00`000000oooo
0?ooo`090?ooo`001`3oool00`000000oooo0?ooo`0n0?ooo`03003ogP3oool0oooo0080oooo00<0
0?oN0?ooo`3oool02P3oool00`00onH0oooo003oiP0@0?ooo`030000003oool0oooo00T0oooo00<0
0?oN0?ooo`3oool0203oool00`00oeX00?oN0?ooo`050?ooo`<00000503oool00`00obT0oooo003o
:@030?ooo`03003ogP3oool0oooo00<0oooo00@0o`3V0?ooo`3oool00?oN303oool00`00omh0oooo
0?ooo`030?ooo`0300000000onH0oooo00@0oooo00<00?oV0?ooo`00onH0203oool00`00omh0oooo
0?ooo`080?ooo`03003ogP3oool00?oN01P0oooo00<000000?ooo`3oool02@3oool000L0oooo00<0
00000?ooo`3oool0?P3oool00`00omh0oooo0?ooo`0?0?ooo`03003ogP3oool00?oN0100oooo00<0
00000?ooo`3oool02@3oool00`00omh0oooo0?ooo`030?ooo`03003oiP3oool0oooo0080oooo00<0
0?mJ003oiP3oool00P3oool3000001H0oooo00@00?oN003o:@3oool00?lY203oool01@00omh0oooo
0?ooo`3oool00?oN00`0oooo00<00?oV003ogP3oool0103oool00`000000oooo0?ooo`030?ooo`03
003ogP3oool0oooo00P0oooo00<00?oN0?ooo`3oool0203oool00`00omD0oooo0?ooo`020?ooo`03
003ogP3oool0oooo01<0oooo00<000000?ooo`3oool02@3oool000L0oooo00<000000?ooo`3oool0
B`3oool00`00omh0oooo0?ooo`0<0?ooo`03003oiP3oool00?oV00H0oooo00<000000?ooo`3oool0
1@3oool00`00omh0oooo0?ooo`060?ooo`03003oFP00omh0oooo00<0oooo00<0U03o0?ooo`000000
0P0000020?ooo`03003ogP3oool0oooo01@0oooo00D00?oN0?l0a@3oool0oooo0?l0c@070?ooo`03
003ogP3oool0oooo00l0oooo00<00?oV0?ooo`3oool0103oool00`000000oooo0?ooo`040?ooo`04
003oiP3oool0oooo003ogQD0oooo00<00?oN0?ooo`3oool04`3oool00`000000oooo0?ooo`090?oo
o`001P3oool00`000000oooo0?ooo`1<0?ooo`03003ogP3oool0oooo00`0oooo00<00?oN0?ooo`00
omh01P3oool00`000000oooo0?ooo`050?ooo`03003ogP3oool0oooo00H0oooo00@0U03o0?ooo`3o
ool0oooo100000000`08o`000?oV0?ooo`0J0?ooo`050?l0a@3oool0oooo0?l0c@00omh02P3oool0
0`00omh0oooo0?ooo`0C0?ooo`070000003oool0oooo0?ooo`00omh0oooo003ogP020?ooo`03003o
gP3oool0oooo02T0oooo00<000000?ooo`3oool02@3oool000H0oooo00<000000?ooo`3oool0BP3o
ool00`00omh0oooo003ogP090?ooo`03003ogP3oool00?oN00/0oooo00<000000?ooo`3oool01`3o
ool00`00oe80oooo0?ooo`040?ooo`0509`0o`000000000000000000onH00`3oool00`3o0200oooo
0?ooo`0N0?ooo`03003ogP3oool0oooo00@0oooo00D00?oN0?ooo`3oool0oooo003ogP0F0?ooo`04
0000003oool0oooo003oiPl0oooo00<00?oV0?ooo`3oool02@3oool00`00onH0oooo0?ooo`0B0?oo
o`030000003oool0oooo00T0oooo00060?ooo`030000003oool0oooo04X0oooo00<00?oN0?ooo`00
omh02@3oool00`00omh0oooo003ogP050?ooo`03003oiP3oool00?oV00<0oooo00<000000?ooo`3o
ool01`3oool00`00oeX0oooo0?ooo`020?ooo`<000000P3oool00`0@o`000?oN0?ooo`020?ooo`03
0?l0803oool0oooo02D0oooo00<00?oN0?ooo`3oool0603oool00`000000oooo0?ooo`030?ooo`03
003ogP3oool0oooo00X0oooo00<00?oN0?ooo`3oool02@3oool01000omh0oooo0?ooo`00omhB0?oo
o`030000003oool0oooo00P0oooo00060?ooo`030000003oool0oooo04D0oooo00<00?oV0?ooo`3o
ool0103oool00`00omh0oooo0?ooo`020?ooo`03003oiP3oool0oooo0080oooo00<00?oN0?ooo`3o
ool01@3oool00`00omh0oooo003ogP030?ooo`030000003oool0oooo00L0oooo00<0W03o0?ooo`00
00000P0000020?ooo`04003ogP3oool0oooo0?l062D0oooo00<00?oN0?ooo`3oool03@3oool00`00
omh0oooo0?ooo`0:0?ooo`03003ogP3oool0oooo00<0oooo00<00000003oe@3oool00P3oool01@00
omD0oooo0?ooo`3oool00?oN01L0oooo00<00?oN0?ooo`3oool0403oool00`000000oooo0?ooo`08
0?ooo`001P3oool00`000000oooo0?ooo`150?ooo`03003ogP3oool0oooo00T0oooo00<00?oN0?oo
o`3oool01@3oool00`00omh0oooo0?ooo`080?ooo`030000003oool0oooo00H0oooo0`0000000`0@
o`000?oN0?ooo`050?ooo`050?l0603oool0oooo0?ooo`00onH08@3oool00`00omh0oooo0?ooo`09
0?ooo`05003ogP3oool0oooo0?ooo`00omh0303oool00`00omD0oooo0?ooo`040?ooo`030000003o
ool0oooo00<0oooo00<00?oN0?ooo`00omh0203oool00`00omh0oooo0?ooo`0O0?ooo`030000003o
ool0oooo00P0oooo00060?ooo`030000003oool0oooo04`0oooo00<00?oN0?ooo`3oool02P3oool0
0`00onH0oooo0?ooo`050?ooo`04003oiP3oool0oooo003ogP@0oooo100000030?ooo`030?l0203o
ool0oooo00T0oooo00<00?oN0?ooo`3oool00`3oool00`00oaP00?oN0?ooo`0U0?ooo`03003ogP3o
ool0oooo00D0oooo00<00?oN0?ooo`3oool01`3oool00`00omD0oooo0?ooo`040?ooo`030000003o
ool0oooo0080oooo00<00?oN0?ooo`3oool0203oool00`00onH0oooo0?ooo`0O0?ooo`030000003o
ool0oooo00P0oooo00050?ooo`030000003oool0oooo02P0oooo00<00?oE0?ooo`3oool08P3oool0
0`00omh0oooo0?ooo`0>0?ooo`05003ogP3oool0oooo0?ooo`00omh00P3oool00`00onH00?oN0000
0002000000L0oooo00<0o`080?ooo`3oool02P3oool00`00oa00oooo0?ooo`020?ooo`03003o6000
omD0oooo01h0oooo00<00?oN0?ooo`3oool0203oool01@00omh0oooo003ogP3oool00?oN00T0oooo
00<00?oN0?ooo`3oool01@3oool00`000000oooo0?ooo`040?ooo`04003ogP3oool0oooo003ogPL0
oooo00<00?oN0?ooo`3oool02@3oool00`00omh0oooo0?ooo`0@0?ooo`030000003oool0oooo00P0
oooo00050?ooo`030000003oool0oooo02P0oooo00<00?oN0?ooo`3oool09P3oool01000omh0oooo
0?ooo`00onH70?ooo`03003oiP3oool00?oV00D0oooo0`00000G0?ooo`03003o603oool0oooo0080
oooo00<0o`2D0?ooo`3oool07P3oool00`00onH0oooo0?ooo`040?ooo`07003ogP3oool0oooo0?oo
o`00omh0oooo003ogP0C0?ooo`030000003oool0oooo0080oooo00<00?oV0?ooo`00onH00P3oool0
0`00omh0oooo0?ooo`050?ooo`03003ogP3oool0oooo00T0oooo00<00?oN0?ooo`3oool0403oool0
0`000000oooo0?ooo`080?ooo`001@3oool00`000000oooo0?ooo`0[0?ooo`03003ogP3oool0oooo
02<0oooo00@00?oN0?ooo`3oool00?oN0P3oool00`00onH0oooo0?ooo`020?ooo`05003ogP3oool0
0?oN003oiP3oool00`0000000`00omh000000?ooo`0@0?ooo`03003oiP3oool0oooo00@0oooo00<0
o`2;0?ooo`3oool00P3oool00`3o09@0oooo0?ooo`0U0?ooo`03003ogP3oool0oooo01P0oooo00D0
00000?ooo`3oool0oooo003ogP040?ooo`03003ogP3oool0oooo01D0oooo00<00?oV0?ooo`3oool0
303oool00`000000oooo0?ooo`080?ooo`001@3oool00`000000oooo0?ooo`0[0?ooo`03003oe@3o
ool0oooo01h0oooo00<00?oV0?ooo`3oool01P3oool00`00omh0oooo003ogP060?ooo`<0000000D0
0?oN0?ooo`3oool0oooo0000000@0?ooo`03003o4000omh0oooo00D0oooo00<0o`2D0?ooo`3oool0
8`3oool00`00onH0oooo0?ooo`0P0?ooo`030000003oool0oooo00<0oooo00<00?oN0?ooo`00omh0
203oool00`00onH0oooo0?ooo`0<0?ooo`03003ogP3oool0oooo00d0oooo00<000000?ooo`3oool0
1`3oool000D0oooo00<000000?ooo`3oool0C03oool00`00omh0oooo0?ooo`060?ooo`03003ogP3o
ool0oooo0080oooo10000000103oool00?oV0?ooo`00onH30?ooo`030000003oool0oooo00h0oooo
00<0o`230?ooo`3oool0:`3oool00`00omh0oooo0?ooo`0>0?ooo`03003ogP3oool0oooo00/0oooo
00<00?oN0?ooo`3oool00P3oool00`000000oooo003ogP020?ooo`03003ogP3oool0oooo0080oooo
00<00?oN0?ooo`3oool00`3oool00`00omh0oooo0?ooo`0L0?ooo`030000003oool0oooo00L0oooo
00050?ooo`030000003oool0oooo0340oooo00<00?oN0?ooo`3oool08P3oool01@00onH000000000
000000000?oV00@0oooo00<00?oN0?ooo`3oool00`3oool00`000000oooo0?ooo`0>0?ooo`050?l0
P`3oool0oooo0?ooo`00onH0=P3oool01@00omh0oooo0?ooo`3oool00?oN00d0oooo00<00?oN0?oo
o`3oool00`3oool00`0000000?oN0?ooo`040?ooo`03003oiP3oool00?oN00T0oooo00<00?oV0?oo
o`3oool0203oool00`00omh0oooo0?ooo`0=0?ooo`030000003oool0oooo00L0oooo00050?ooo`03
0000003oool0oooo02l0oooo00<00?oN0?ooo`00omh08P3oool3000000040?ooo`00omh0oooo003o
gPL0oooo00@00?oN0?ooo`00omh000001P3oool00`00omD0oooo0?ooo`0;0?ooo`03003ogP3oool0
oooo00L0oooo00<00?oE0?ooo`3oool0503oool00`00omh0oooo0?ooo`0C0?ooo`03003ogP3oool0
oooo01H0oooo00@000000?ooo`3oool00?oN0P3oool01@00omh0oooo0?ooo`3oool00?oN00L0oooo
00<00?oN0?ooo`3oool0203oool01000onH0oooo0?ooo`00omh<0?ooo`030000003oool0oooo00L0
oooo00040?ooo`030000003oool0oooo02X0oooo00<00?oN0?ooo`3oool00`3oool2003ogR00oooo
0`0000000`00omh0oooo0?ooo`0;0?ooo`05003oiP3oool00?oV00000000onH00`3oool00`00omh0
oooo003ogP0G0?ooo`03003ogP3oool0oooo01@0oooo00<00?oN0?ooo`3oool0303oool00`00onH0
oooo0?ooo`080?ooo`03003ogP3oool00?oN0080oooo00<00?oN0?ooo`3oool0203oool00`00omh0
oooo0?ooo`020?ooo`040000003oool0oooo003ogPH0oooo00<00?oN0?ooo`3oool04`3oool00`00
onH0oooo0?ooo`0:0?ooo`030000003oool0oooo00L0oooo00040?ooo`030000003oool0oooo02X0
oooo00<00?oN0?ooo`3oool0103oool00`00omD00?oN0?ooo`0J0?ooo`@0000000<00?oN0?ooo`3o
ool00`3oool01000omh0oooo0?ooo`00omh20?ooo`03003oiP3oool0oooo00@0oooo0`00omh30?oo
o`03003oe@3oool0oooo0100oooo00<00?oV0?ooo`3oool05P3oool00`00omD0oooo0?ooo`0A0?oo
o`03003ogP3oool0oooo00@0oooo00X00?oN0?ooo`3oool0oooo003ogP3oool00?oN0?ooo`00onH0
0?oN0P3oool00`00omh0oooo0?ooo`050?ooo`03003ogP3oool0oooo00<0oooo00<000000?ooo`3o
ool0103oool00`00omh0oooo003ogP0R0?ooo`030000003oool0oooo00L0oooo00040?ooo`030000
003oool0oooo0300oooo00<00?oN0?ooo`00omh0603oool3000000X0oooo00@00?oN0?ooo`3oool0
0?oN0P3oool00`00omh0oooo003ogP040?ooo`03003oiP00omh0oooo01H0oooo00<00?oN003oiP3o
ool05P3oool00`00omh0oooo0?ooo`0H0?ooo`03003ogP3oool0oooo00D0oooo00D00?oN0?ooo`3o
ool0oooo003ogP0>0?ooo`030000003oool0oooo00<0oooo00<00?oV0?ooo`00onH01`3oool00`00
onH0oooo003oiP0;0?ooo`03003oiP3oool00?oV00X0oooo00<000000?ooo`3oool01`3oool000@0
oooo00<000000?ooo`3oool02`3oool00`00omD0oooo0?ooo`0L0?ooo`03003ogP3oool0oooo00<0
oooo00<00?oN0?ooo`3oool05@3oool300000140oooo00D00?oV0?ooo`3oool00?oN003oiP050?oo
o`030000003oool0oooo01H0oooo00<00?oN0?ooo`3oool03P3oool00`00omh0oooo0?ooo`0H0?oo
o`03003ogP3oool0oooo00L0oooo00<00?oN003oiP3oool00`3oool00`00omh0oooo003ogP0:0?oo
o`07003ogP3oool0oooo0?ooo`00oi@0oooo0000000>0?ooo`03003ogP3oool00?oN00/0oooo00<0
0?oN0?ooo`00omh02`3oool00`000000oooo0?ooo`060?ooo`00103oool00`000000oooo0?ooo`08
0?ooo`04003ogP3oool0oooo003ogQ`0oooo00<00?oN0?ooo`00omh03`3oool00`00omh0oooo0?oo
o`080?ooo`<000004`3oool2003ogP80oooo00<00?oV0?ooo`3oool0103oool00`000000oooo0?oo
o`0W0?ooo`03003ogP3oool0oooo01P0oooo00<00?oV0?ooo`3oool01@3oool01000omh0oooo003o
gP00omh40?ooo`03003ogP3oool00?oN00X0oooo00D00?oN0?ooo`3oool0oooo003oU0020?ooo`09
0000003oool00?oN0?ooo`00onH0oooo003oiP3oool00?oV01H0oooo00<00?oN0?ooo`3oool0203o
ool00`000000oooo0?ooo`060?ooo`00103oool00`000000oooo0?ooo`080?ooo`03003ogP3oool0
oooo0080oooo00<00?oE0?ooo`3oool0603oool00`00omh0oooo0?ooo`0=0?ooo`03003ogP3oool0
0?oN00L0oooo0`00000B0?ooo`05003ogP3oool0oooo0?ooo`00omh02P3oool00`000000oooo0?oo
o`160?ooo`05003ogP3oool0oooo0?ooo`00omh01P3oool00`00onH0oooo0?ooo`030?ooo`03003o
gP3oool0oooo00T0oooo00`002co0?ooo`3oool000000?ooo`00omh0oooo003ogP3oool00?oN0?oo
o`00omh20?ooo`03003ogP3oool0oooo00<0oooo00<00?oV0?ooo`00onH02P3oool00`00omh00?oV
0?ooo`090?ooo`030000003oool0oooo00H0oooo00030?ooo`030000003oool0oooo00<0oooo00<0
0?oN0?ooo`3oool01P3oool00`00omh0oooo003ogP0L0?ooo`03003ogP3oool0oooo00/0oooo00<0
0?oN0?ooo`3oool00`3oool4000001D0oooo00<00?oV0?ooo`3oool01P3oool3003ogP<0oooo00<0
00000?ooo`3oool0103oool00`00omh0oooo0?ooo`0o0?ooo`03003oiP3oool0oooo00@0oooo00D0
0?oN0?ooo`3oool0oooo003ogP020?ooo`04003ogP3oool0oooo003ogP/0oooo00D002So0?ooo`3o
ool0oooo000000040?ooo`03003oiP3oool0oooo0080oooo00<00?oN0?ooo`3oool00`3oool00`00
omh0oooo003ogP0:0?ooo`03003oiP3oool0oooo00T0oooo00<000000?ooo`3oool01P3oool000<0
oooo00<000000?ooo`3oool00`3oool00`00omh0oooo0?ooo`060?ooo`03003oe@3oool0oooo01`0
oooo00<00?oN0?ooo`3oool01@3oool00`00omh0oooo0?ooo`050?ooo`04003ogP00000000000000
0280oooo00<00?oV003ogP00onH00`3oool00`000000oooo0?ooo`020?ooo`03003oe@3oool00?oE
03@0oooo00<00?oN0?ooo`3oool04@3oool00`00onH0oooo0?ooo`040?ooo`03003ogP3oool00?n;
0140oooo00@000000?ooo`3oool00?oN0`3oool00`00onH0oooo003oiP080?ooo`03003ogP3oool0
oooo00T0oooo00<00?oN0?ooo`3oool01`3oool00`000000oooo0?ooo`060?ooo`000`3oool00`00
0000oooo0?ooo`060?ooo`03003ogP3oool0oooo02X0oooo00<00?oN0?ooo`3oool00P3oool01`00
omh000000000000000000?oN0?ooo`00omh07@3oool00`00onH0oooo0?ooo`080?ooo`03003ogP3o
ool0oooo0080oooo00<00?oN0?ooo`3oool0<03oool01@00omh0oooo0?ooo`3oool00?oN00`0oooo
00<00?oV0?ooo`3oool0203oool00`00omh0oooo0?ooo`020?ooo`03003oU03oool0oooo0080oooo
00<00?oN0?ooo`3oool02`3oool02@000000oooo003ogP3oool00?oN0?ooo`00omh0oooo003ogP07
0?ooo`03003ogP00onH0oooo00X0oooo00<00?oV0?ooo`3oool01`3oool00`000000oooo0?ooo`06
0?ooo`000`3oool00`000000oooo0?ooo`060?ooo`03003oe@3oool0oooo02d0oooo0`0000030?oo
o`03003ogP3oool0oooo0080oooo00<00?oN0?ooo`3oool0603oool00`00omh0oooo0?ooo`030?oo
o`04003ogP3oool00?oN0?ooo`800?oN7P3oool00`00omD0oooo0?ooo`0F0?ooo`03003ogP3oool0
oooo00h0oooo00<00?oN0?ooo`3oool0203oool00`00onH0oooo0?ooo`020?ooo`03000Ho`3oool0
oooo0080oooo00<00?oN0?ooo`3oool01P3oool00`00omh0oooo0?ooo`030?ooo`090000003oool0
oooo003oiP3oool00?oV003ogP3oool00?oN00H0oooo00<00?oV0?ooo`3oool0503oool00`000000
oooo0?ooo`060?ooo`000`3oool00`000000oooo0?ooo`0B0?ooo`03003ogP3oool00?oN01h0oooo
0`0000000`00omh0oooo0?ooo`050?ooo`800?oN00<0oooo003ogP3oool07`3oool01P00omh0oooo
003oiP3oool00?oV003ogQh0oooo00<00?oN0?ooo`3oool03P3oool00`00omh0oooo0?ooo`030?oo
o`03003ogP3oool0oooo00H0oooo00<00?oN0?ooo`3oool02`3oool00`00oh/00?oN003oR`050?oo
o`07003ogP3oool0oooo0?ooo`007?l0oooo003oiP0;0?ooo`03003ogP3oool0oooo00<0oooo00@0
00000?ooo`3oool00?oN0P3oool00`00onH00?oN003oiP070?ooo`03003oiP3oool0oooo00d0oooo
00<00?oN0?ooo`3oool00`3oool00`000000oooo0?ooo`060?ooo`000`3oool00`000000oooo0?oo
o`0B0?ooo`03003oe@3oool00?oE01<0oooo00<00?mJ0?ooo`3oool0103oool400000003003ogP3o
ool0oooo00D0oooo00<00?oN0?ooo`3oool00P00omhO0?ooo`03003oiP3oool0oooo00<0oooo00<0
0?oV00000000onH0;`3oool00`00omh0oooo0?ooo`030?ooo`05003ogP3oool0oooo0?ooo`00omh0
103oool00`00omh0oooo0?ooo`0;0?ooo`03003oP`00onH00?n;00<0oooo00T00?oN0?ooo`00omh0
oooo003ogP3oool00?oN0?ooo`00omh00`3oool00`00omh0oooo003ogP0:0?ooo`09003o]03oool0
00000?ooo`00omh0oooo003oiP3oool00?oV00P0oooo00<00?oN0?ooo`3oool03@3oool00`00omD0
oooo0?ooo`040?ooo`030000003oool0oooo00D0oooo00020?ooo`030000003oool0oooo00d0oooo
00<00?oN0?ooo`3oool06@3oool0102D0?l0oooo0?ooo`3oool300000080oooo00<00?oN0?ooo`3o
ool00P3oool01P00omh0oooo0?ooo`00omh0oooo003ogPD0oooo00<00?oN0?ooo`3oool06`3oool0
0`00omh00?oV0?ooo`030?ooo`04003ogP000000oooo003oiRl0oooo00<00?oN0?ooo`3oool01P3o
ool00`00omh0oooo0?ooo`040?ooo`03003ogP3oool0oooo00T0oooo00@000co0?ooo`3oool000co
0P3oool01@00omh0oooo003oiP3oool00?oV00@0oooo00H00?oN0?ooo`3oool00?oN0?ooo`00omh:
0?ooo`04003o]03oool0oooo00000080oooo00@00?oN0?ooo`3oool00?oV0`3oool00`00omh0oooo
0?ooo`0H0?ooo`030000003oool0oooo00D0oooo00020?ooo`030000003oool0oooo00d0oooo00<0
0?oE0?ooo`3oool06@3oool0102;0?l0000000000000000:0?ooo`04003ogP3oool0oooo003ogP@0
oooo00@00?oN0?ooo`3oool00?oN7P3oool00`00omh0oooo0?ooo`030?ooo`030000003oool00?oN
02l0oooo00<00?oE0?ooo`00omh03@3oool00`00omh0oooo0?ooo`090?ooo`050008o`00onH0oooo
000@o`00omh01P3oool01`00omh0oooo0?ooo`3oool00?oN0?ooo`00onH03@3oool01@00VOl0oooo
0?ooo`3oool000000080oooo00<00?oN0?ooo`00omh00`3oool00`00omh0oooo0?ooo`060?ooo`03
003ogP3oool0oooo00T0oooo00<00?oN0?ooo`3oool00`3oool00`000000oooo0?ooo`050?ooo`00
0P3oool00`000000oooo0?ooo`090?ooo`03003ogP3oool00?oN01/0oooo0`0000020?ooo`030?l0
803oool0oooo0100oooo00<00?oN0?ooo`3oool01P3oool00`00omh0oooo0?ooo`0L0?ooo`030000
003oool0oooo02P0oooo00<00?oN0?ooo`3oool01P3oool00`00omh0oooo0?ooo`070?ooo`03003o
gP3oool0oooo00h0oooo00@00?oN0?ooo`3oool00?oV103oool00`00omh00?oV003ogP030?ooo`03
003ogP3oool00?oN00@0oooo00<00?oN0?ooo`3oool01P3oool02000VOl0oooo0?ooo`00olD0oooo
0000003oool00?oV3P3oool00`00omh0oooo0?ooo`090?ooo`03003ogP3oool0oooo00<0oooo00<0
00000?ooo`3oool01@3oool00080oooo00<000000?ooo`3oool02@3oool00`00omD0oooo003oe@0H
0?ooo`<000001@3oool00`3o02T0oooo0?ooo`0;0?ooo`03003ogP3oool0oooo00P0oooo00@00?oN
0?ooo`3oool00?oN7P3oool00`000000oooo003ogP0X0?ooo`03003ogP3oool0oooo0100oooo00<0
0?oN0?ooo`3oool05P3oool2003ogP80oooo00<00?oV0?ooo`3oool00P3oool01@00oj`0oooo0?oo
o`3oool00?oN00P0oooo00@00?oN0?ooo`3oool00?o50P3oool00`000000oooo0?ooo`050?ooo`03
003ogP3oool0oooo01L0oooo00<000000?ooo`3oool01@3oool00080oooo00<000000?ooo`3oool0
0`3oool00`00omh0oooo0?ooo`050?ooo`04003ogP3oool0oooo003ogQ40oooo1000000F0?ooo`03
003ogP3oool0oooo00P0oooo00<00?oN0?ooo`3oool07`3oool00`000000oooo003oiP030?ooo`03
003ogP3oool00?oN02h0oooo00<00?oN0?ooo`3oool0603oool01000onH0oooo0?ooo`00omh50?oo
o`06003ogP00onH0oooo0?ooo`00omh0097o303oool01000omh0oooo0?ooo`00c_l20?ooo`040000
003oool0oooo003ogP80oooo00<00?oN0?ooo`00omh01`3oool00`00omh0oooo0?ooo`0?0?ooo`03
0000003oool0oooo00D0oooo00020?ooo`030000003oool0oooo00<0oooo00<00?oE0?ooo`3oool0
1@3oool01000omD0oooo0?ooo`00omh>0?ooo`<000004P3oool00`00oaP0oooo0?ooo`0;0?ooo`03
003ogP3oool0oooo00D0oooo00<00?oN0?ooo`3oool0703oool00`000000oooo0?ooo`030?ooo`03
003oe@3oool00?oE02h0oooo00<00?n30?ooo`3oool0103oool00`00omh0oooo0?ooo`0A0?ooo`09
003ogP3oool0oooo003oiP3oool00?nd0?ooo`00omh0oooo00800?oN0P3oool01`00onH0097o0?oo
o`3oool00?oN0?ooo`00omh02P3oool01`00b_l0oooo003oc@3oool000000?ooo`00omh00P3oool0
0`00omh0oooo0?ooo`070?ooo`03003ogP3oool0oooo00l0oooo00<000000?ooo`3oool01@3oool0
0080oooo00<000000?ooo`3oool01@3oool00`00omh0oooo0?ooo`0B0?ooo`<000005@3oool00`00
oaP0oooo0?ooo`0;0?ooo`03003ogP3oool0oooo0080oooo00@00?oN0?ooo`3oool00?oN7P3oool0
0`0000000?oN0?ooo`060?ooo`03003ogP3oool0oooo02H0oooo00<00?n30?ooo`3oool00P3oool0
0`00oh<0oooo0?ooo`040?ooo`03003ogP3oool0oooo00L0oooo00<00?oN0?ooo`3oool02P3oool0
1`00oj`00?oV003o[03oool00?oN0?ooo`00omh01@3oool01@00omh0oooo003oa@3oool00?oN00`0
oooo00@00?oE0?ooo`3oool00000803oool00`000000oooo0?ooo`040?ooo`0000<0oooo0000003o
ool01`3oool00`00omD0oooo0?ooo`0?0?ooo`<00000603oool00`3o09`0oooo0?ooo`0@0?ooo`03
003ogP3oool0oooo01l0oooo00<00000003ogP3oool01P3oool00`00omh0oooo0?ooo`0V0?ooo`03
003oN`3oool0oooo0080oooo00<0203o0?ooo`00omh02@3oool00`00omh0oooo0?ooo`020?ooo`03
003ogP3oool0oooo00X0oooo00<008Wo003ogP00SOl00`3oool00`00onH00?oN0?ooo`030?ooo`03
003oiP3oool00<Ko00D0oooo00<00?oN0?ooo`3oool01P3oool01@00n_l0oooo0?ooo`3oool00000
00<0oooo00<00?oE0?ooo`00omD00P3oool00`00omh0oooo0?ooo`0D0?ooo`030000003oool0oooo
00@0oooo00000`3oool000000?ooo`0<0?ooo`03003oJP3oool0oooo00L0oooo0`00000K0?ooo`03
0?l0W03oool0oooo00/0oooo00<00?oN0?ooo`3oool02@3oool00`00omh0oooo003ogP0H0?ooo`03
0000003oool0oooo02P0oooo00<00?mk0?ooo`3oool0103oool00`0@0?l0oooo0?ooo`020?ooo`03
00P0o`3oool00?oN00H0oooo00@00?oN0?ooo`3oool00?oN4@3oool01@00ROl0oooo0029o`3oool0
0?nm0080oooo00<00?oV0?ooo`3oool0103oool00`00a_l0oooo003oc@030?ooo`03003ogP3oool0
oooo00H0oooo00<00?[o0?ooo`3oool00P3oool01P000000oooo0?ooo`00omh0oooo003ogP80oooo
00<00?oN0?ooo`3oool0503oool00`000000oooo0?ooo`040?ooo`0000<0oooo0000003oool02P3o
ool00`00ofX0oooo003oJP050?ooo`@0000000<00?oN0?ooo`3oool0:@3oool00`00omh0oooo0?oo
o`090?ooo`03003ogP3oool00?oN01P0oooo00<000000?ooo`3oool01P3oool00`00omh0oooo0?oo
o`0O0?ooo`03003oP`3oool0oooo00@0oooo00<0203o0?ooo`3oool01`3oool01@00omh0oooo0?oo
o`3oool00?oN01D0oooo00@00?oN003oa@3oool00?o52`3oool2003oeA00oooo00<000000?ooo`3o
ool0703oool00`000000oooo0?ooo`040?ooo`0000<0oooo0000003oool02P3oool00`00ofX0oooo
05X0o`020?ooo`<0000000<0oooo003ogP3oool0<`3oool00`00omh0oooo0?ooo`0N0?ooo`030000
003oool0oooo00H0oooo00D00?oN0?ooo`3oool0oooo003ogP0M0?ooo`030100o`3oool0oooo00@0
oooo00<00?oN0?ooo`3oool01`3oool00`00omh0oooo0?ooo`0G0?ooo`05003oiP00`_l0oooo0?oo
o`00`_l00`3oool01@00omh0oooo003ogP3oool00?oN0080oooo00<00?Ko003ogP3oool02`3oool0
0`00omh0oooo0?ooo`020?ooo`040000003oool0oooo003ogQX0oooo00<000000?ooo`3oool0103o
ool000030?ooo`000000oooo00@0oooo00<00?mc0?ooo`3oool00P3oool00`1B0?l0oooo0?ooo`03
00000003003o403oool0oooo03H0oooo00<00?oN0?ooo`3oool00`3oool00`00omh0oooo003ogP0H
0?ooo`030000003oool0oooo00X0oooo00<00?oV0?ooo`3oool06`3oool00`0@0?l0oooo0?ooo`0>
0?ooo`03003ogP3oool0oooo00@0oooo00<00?oN0?ooo`3oool04@3oool03000`_l0oooo0?ooo`00
`_l0oooo003oc@3oool00?oN0?ooo`00omh0oooo003oe@80oooo00<00?Ko0?ooo`3oool02`3oool0
2@00omh0oooo0?ooo`3oool00?o^0?ooo`000000oooo003ogP070?ooo`03003ogP3oool0oooo0100
oooo00<000000?ooo`3oool0103oool000030?ooo`000000oooo00@0oooo00<00?mc0?ooo`3oool0
0P3oool300000004003o6000omh0oooo0?l0Rch0oooo00<00?oN0?ooo`00omh0603oool00`000000
oooo0?ooo`0a0?ooo`03003ogP3oool0oooo00L0oooo00<00?n/0?ooo`3oool00P3oool00`00omh0
oooo0?ooo`070?ooo`03003ogP3oool0oooo00X0oooo00<00?oE003ogP00m_l02@3oool01@00omh0
oooo0?ooo`3oool00?oV00`0oooo00@00?nT0?ooo`3oool00000203oool00`00omh0oooo0?ooo`0@
0?ooo`030000003oool0oooo00@0oooo00000`000000oooo0?ooo`030?ooo`030440o`3oool0oooo
00<000000`3oool0103o09@0oooo0?ooo`3o08<h0?ooo`03003ogP3oool0oooo00`0oooo00<00?oN
0?ooo`3oool03`3oool00`000000oooo0?ooo`0:0?ooo`03003oiP3oool0oooo02@0oooo00<00?oN
0?ooo`3oool00P3oool00`00oj`0oooo0?ooo`020?ooo`03003o[03oool0oooo00P0oooo00D00?oN
0?ooo`3oool0oooo003ogP0<0?ooo`03003bo`00omh00?Ko00D0oooo00T00?oE0?ooo`3oool0oooo
003oe@3oool00?oN0?ooo`00omh0303oool01@00oj@0oooo0?ooo`3oool0000001X0oooo00<00000
0?ooo`3oool0103oool000030000003oool0oooo0080oooo100000000`00oaP0oooo0?ooo`030?oo
o`030?l0R`3oool0oooo03T0oooo00<00?oN0?ooo`3oool02P3oool00`00onH0oooo003oiP0A0?oo
o`030000003oool0oooo00X0oooo00<00?oN0?ooo`3oool09`3oool00`00omh0oooo003o[0040?oo
o`030021o`3oool0oooo00P0oooo00<00?oN0?ooo`3oool03P3oool00`00m_l0oooo0?ooo`050?oo
o`03003ogP3oool0oooo00<0oooo00<00?oV0?ooo`00ooH00`3oool00`00omh0oooo0?ooo`0:0?oo
o`030000003oool0oooo01T0oooo00<000000?ooo`3oool00`3oool000030000003oool000000080
0000103oool00`3o0:`0oooo0?ooo`090?ooo`04003o<@3oool0oooo003o<Cl0oooo00<00?oN0?oo
o`3oool04@3oool00`000000oooo0?ooo`0@0?ooo`03003oiP3oool0oooo01`0oooo00<00?nT0?oo
o`3oool00P3oool00`00omh0oooo001mo`040?ooo`05001mo`3oool0oooo003o_@00omh06@3oool0
0`00omD0oooo0?ooo`0;0?ooo`05003okP3oool0oooo0?ooo`00omh0203oool00`00omh0oooo0?oo
o`020?ooo`030000003oool0oooo01P0oooo00<000000?ooo`3oool00`3oool0008000001`3oool0
0`3o0:@0oooo0?ooo`090?ooo`04003o<@3oool0oooo0?l0kST0oooo00<00?oV0?ooo`3oool01@3o
ool00`00onH0oooo0?ooo`0?0?ooo`030000003oool0oooo0100oooo00<00?oN0?ooo`3oool0703o
ool00`00oj`0oooo0?ooo`040?ooo`04001mo`3oool0oooo003oa@@0oooo00<00;[o003ogP3oool0
103oool00`00omh0oooo0?ooo`0A0?ooo`03003ogP3oool0oooo0080oooo00<00?o^0?ooo`3oool0
0P3oool01@00omh0oooo0?ooo`3oool00?nT00`0oooo00<00?oN0?ooo`3oool00`3oool00`000000
oooo0?ooo`0G0?ooo`030000003oool0oooo00<0oooo00000`3oool000000000000<0?ooo`03003o
>@3oool0oooo00<0oooo00@0o`3f0?ooo`3oool0o`3V>@3oool00`00omh0oooo0?ooo`030?ooo`04
003ogP3oool00?oN003ogQ00oooo00<000000?ooo`3oool0;`3oool00`00NOl0oooo0?ooo`070?oo
o`03003o_@3oool0oooo0080oooo00<00;[o0?ooo`00old0103oool00`00omh0oooo0?ooo`0D0?oo
o`05003omP3oool00?of0?ooo`00omh00P3oool02@00omh0oooo0?ooo`3oool00?nL0?ooo`00omh0
oooo003ogP0=0?ooo`05003^o`3oool000000?ooo`00omh01P3oool00`00omh0oooo0?ooo`0=0?oo
o`030000003oool0oooo00<0oooo00030?ooo`030000003oool0oooo00T0oooo00<00?li0?ooo`3o
ool00`3oool00`3o0?H0oooo0?ooo`100?ooo`05003oiP3oool0oooo003ogP00onH00P3oool00`00
omh0oooo0?ooo`0:0?ooo`030000003oool0oooo00l0oooo00D00?oV0?ooo`3oool0oooo003oiP0K
0?ooo`04001mo`3oool0oooo003o_@H0oooo00<00;[o0?ooo`3oool0103oool00`00old0oooo0?oo
o`0=0?ooo`03003ogP3oool0oooo00T0oooo00D00?o^0?ooo`00oi`0oooo003ogP080?ooo`03003o
iP3oool00?oV00d0oooo00D00?;o0?ooo`3oool00000003oiP060?ooo`03003ogP3oool0oooo00d0
oooo00<000000?ooo`3oool00`3oool000@0oooo00<000000?ooo`3oool0203oool00`3f0?l0oooo
0?ooo`0T0?ooo`03003oe@3oool0oooo01T0oooo00<00?oN0?ooo`3oool01`3oool01000omh0oooo
0?ooo`00omh<0?ooo`030000003oool0oooo00l0oooo00D00?oN0?ooo`3oool0oooo003ogP0N0?oo
o`03003o_@3oool0oooo00@0oooo00<00;Ko0?ooo`00old0103oool00`00l_l0oooo0?ooo`0=0?oo
o`03003ogP3oool0oooo00T0oooo00<00?nL0?ooo`00oi`06P3oool01@00ocT0oooo0?ooo`3oool0
000001H0oooo00<000000?ooo`3oool00`3oool000D0oooo0P0000080?ooo`030?H0o`3oool0oooo
0280oooo00<00?oN0?ooo`00omh06`3oool00`00onH0oooo0?ooo`020?ooo`03003ogP3oool0oooo
00@0oooo0P00omh<0?ooo`030000003oool0oooo0380oooo00<00;Ko0?ooo`3oool0103oool00`00
omh0oooo003bo`040?ooo`03003bo`3oool0oooo00T0oooo00<00?oN0?ooo`3oool03@3oool00`00
oi`0oooo0?ooo`0=0?ooo`03003^o`3oool0oooo0080oooo00<00?oV0?ooo`3oool0103oool00`00
omh00?li0?ooo`020?ooo`030000003oool0oooo00L0oooo00<00?oV0?ooo`3oool02P3oool00`00
0000oooo0?ooo`030?ooo`001`3oool00`000000oooo0?ooo`0Z0?ooo`03003oe@3oool0oooo01`0
oooo00D00?oN0?ooo`3oool0oooo003ogP040?ooo`04003ogP3oool00?oN003ogP`0oooo00<00000
0?ooo`3oool04`3oool01000omh0oooo0?ooo`00onHK0?ooo`03002fo`3oool00?o=00@0oooo00<0
0?oN0?ooo`00k_l02@3oool00`00omh0oooo0?ooo`040?ooo`03003ogP3oool0oooo01d0oooo00<0
0>ko0?ooo`3oool00P3oool00`00omh0oooo0?ooo`040?ooo`03003ogP3oool0oooo00<0oooo00<0
00000?ooo`3oool01P3oool00`00omh0oooo0?ooo`0;0?ooo`030000003oool0oooo0080oooo0008
0?ooo`800000903oool00`00omh0oooo0?ooo`0R0?ooo`03003ogP3oool0oooo00H0oooo00<00?oV
0?ooo`3oool00P3oool01000omh0oooo0?ooo`00onH70?ooo`030000003oool0oooo01<0oooo00D0
0?oV0?ooo`3oool00?oN003oiP0L0?ooo`03003^o`3oool0oooo00h0oooo00<00?oV0?ooo`3oool0
0`3oool00`00omh0oooo0?ooo`0E0?ooo`03003^o`3oool0oooo00H0oooo00D00?la0?ooo`3oool0
oooo003ogP0>0?ooo`030000003oool00?oV01<0oooo00<000000?ooo`3oool00P3oool000X0oooo
00<000000?ooo`3oool08@3oool00`00omD0oooo0?ooo`0V0?ooo`03003ogP3oool0oooo00L0oooo
0P00omh00`3oool00?oN0?ooo`060?ooo`030000003oool0oooo01L0oooo00<00?oN0?ooo`3oool0
6P3oool00`00l_l0oooo0?ooo`0>0?ooo`03003okP00omh0oooo00<0oooo00<00?oN0?ooo`3oool0
1P3oool00`00omh0oooo0?ooo`0:0?ooo`03003Zo`3oool00>[o0080oooo00<00?oN0?ooo`00omh0
0`3oool01@00oc40oooo003ogP3oool00?oN00X0oooo00<00?oN0?ooo`3oool00P3oool00`000000
0?oN0?ooo`060?ooo`03003ogP3oool0oooo00T0oooo00<000000?ooo`3oool00P3oool000/0oooo
0P00001:0?ooo`03003ogP3oool0oooo00<0oooo00<00?oN0?ooo`3oool00P3oool00`00onH0oooo
0?ooo`060?ooo`030000003oool0oooo01L0oooo00<00?oN0?ooo`3oool0:03oool01P00ooH0oooo
0?ooo`3oool00?nL003ogPT0oooo00D00?oN0?ooo`3oool0oooo003ogP0<0?ooo`03003Zo`3oool0
0?lY0080oooo00<00?oN0?ooo`00omh01@3oool00`00omh0oooo0?ooo`0:0?ooo`04003oiP3oool0
oooo003>o`80oooo00<000000?ooo`3oool01@3oool00`00onH0oooo0?ooo`090?ooo`030000003o
ool0oooo0080oooo000=0?ooo`030000003oool0oooo02h0oooo00<00?oN0?ooo`3oool0703oool0
0`00onH0oooo0?ooo`040?ooo`05003ogP3oool0oooo0?ooo`00omh00P3oool00`000000oooo0?oo
o`0G0?ooo`03003oiP00omh0oooo02H0oooo00<00?oN0?ooo`00onh00`3oool00`00oi@0oooo0?oo
o`080?ooo`03003ogP3oool0oooo00h0oooo00<00?lP0?ooo`00ob003`3oool00`00omh0oooo003o
gP080?ooo`0403Wo003oool0oooo000000P0oooo00<00?oN0?ooo`3oool0203oool00`000000oooo
0?ooo`020?ooo`003P3oool00`000000oooo0?ooo`0[0?ooo`03003ogP3oool00?oN02<0oooo00@0
0?oV0?ooo`00onH00?oN0P3oool01000onH0oooo0?ooo`00000J0?ooo`03003oiP3oool0oooo0200
oooo00<00?of0?ooo`3oool00P3oool00`00omh0oooo003oW0070?ooo`03003oiP3oool0oooo01D0
oooo00<00?lY0?ooo`3oool03@3oool01@00onH0oooo003oiP3oool00?oV00L0oooo00<00?oV037o
003oool00P3oool00`000000oooo0?ooo`050?ooo`03003ogP3oool0oooo00P0oooo00<000000?oo
o`3oool00P3oool000l0oooo0P00000[0?ooo`03003ogP3oool0oooo01h0oooo00H00?oV0?ooo`3o
ool00?oN0?ooo`00omh20?ooo`03003oiP3oool00?oN00<0oooo00<000000?ooo`3oool0>`3oool0
0`00ooH0oooo0?ooo`040?ooo`03003oU03oool0oooo00D0oooo00<00?oN0?ooo`3oool0803oool0
0`00omh0oooo0?ooo`020?ooo`03003ogP00b_l0oooo00T0oooo00<00?oN0?ooo`3oool00`3oool0
0`000000oooo003ogP0?0?ooo`030000003oool0oooo0080oooo000A0?ooo`030000003oool0oooo
0280oooo00<00?oN0?ooo`3oool0203oool00`00omh0oooo0?ooo`0>0?ooo`03003oP`3oool0oooo
00D0oooo00@00?oN0?ooo`3oool00?oN0P3oool00`00onH0oooo0?ooo`040?ooo`05003ogP00onH0
0?oN0?ooo`00omh08@3oool00`00omh0oooo0?ooo`0I0?ooo`03003oU03oool0oooo01`0oooo00<0
0?oV0?ooo`3oool03P3oool00`00omh0oooo003ogP050?ooo`0404[o003oool0oooo003oiPT0oooo
00L00?oV0?ooo`3oool00;Ko0?ooo`0000000?oV00l0oooo00<000000?ooo`3oool00P3oool00180
oooo0P00000R0?ooo`03003ogP3oool0oooo00H0oooo00<00?oN0?ooo`00omh0403oool00`000?l0
oooo0?ooo`050?ooo`03003ogP3oool0oooo00P0oooo00P00?oV0?ooo`3oool00?oN003oiP3oool0
0?oV003ogR00oooo00<00?oN0?ooo`3oool06@3oool00`00oi@0oooo0?ooo`0>0?ooo`03003Ro`3o
ool0oooo00<0oooo00D00?oN0?ooo`3oool0oooo003ogP030?ooo`03003ogP3oool0oooo00/0oooo
00@00<;o0?ooo`00`_l00?oN1`3oool0101:o`000?oN0?ooo`00omh90?ooo`04003ogP3oool0oooo
002jo`80oooo00<000000?ooo`3oool03P3oool00`000000oooo0?ooo`010?ooo`00503oool00`00
0000oooo0?ooo`0L0?ooo`03003ogP3oool00?oN00<0oooo00<00?oN0?ooo`3oool00`3oool00`00
omh0oooo0?ooo`020?ooo`03003ogP3oool0oooo00/0oooo00<0203o0?ooo`3oool0203oool01000
omh0oooo0?ooo`00onH20?ooo`03003ogP3oool00?oN00@0oooo00<00?oV00000000onH0B03oool0
0`00i_l0oooo0?ooo`020?ooo`03003o6000omh0oooo00<0oooo00D00?oV0?ooo`3oool0oooo003o
iP0A0?ooo`030036o`3oool00<Ko00T0oooo00@00?oV003ogP3oool00?oN0P3oool00`00onH0oooo
0?ooo`060?ooo`0407_o003oool0oooo00000100oooo00<000000?ooo`3oool00@3oool001D0oooo
00<000000?ooo`3oool06`3oool00`00omh0oooo003ogP030?ooo`03003ogP3oool0oooo00H0oooo
00<00?oN0?ooo`00omh0603oool01000omh0oooo0?ooo`00omh20?ooo`03003oiP3oool00?oV0080
oooo00@00?oN0?ooo`00omh00?oV8@3oool00`00omh0oooo0?ooo`0U0?ooo`03003o603oool0oooo
0080oooo00@00?l@003ogP3oool00?oN3@3oool00`00omh0oooo0?ooo`080?ooo`0305;o003oool0
D_l000X0oooo00<00?oV002^o`00onH00P3oool00`00omh0oooo0?ooo`060?ooo`0708?o003oool0
oooo0?ooo`000000oooo003ogP060?ooo`03003oe@3oool0oooo00@0oooo00<000000?ooo`3oool0
0@3oool001H0oooo0P00000E0?ooo`03003ogP3oool0oooo00`0oooo00<00?oN0?ooo`3oool00`3o
ool00`00omh0oooo0?ooo`0K0?ooo`03003ogP3oool0oooo0080oooo00@00?oN0?ooo`3oool00?oV
0P3oool00`00omh00?oV003ogP0O0?ooo`03003ogP3oool0oooo01h0oooo00<00>Ko0?ooo`3oool0
0`3oool00`00onH00?l@0?ooo`060?ooo`03003ogP3oool0oooo00L0oooo00D00?oN0?ooo`3oool0
oooo003ogP0:0?ooo`0305[o003oool0F_l000L0oooo00D00?oV0?ooo`3oool0oooo002bo`0A0?oo
o`0300000000omh0oooo00D0oooo00<00?oN0?ooo`3oool0103oool00`000000oooo0?ooo`010?oo
o`00603oool00`000000oooo0?ooo`0B0?ooo`03003ogP3oool0oooo00`0oooo00<00?oN0?ooo`3o
ool08@3oool01000omh0oooo0?ooo`00onH60?ooo`04003ogP0000000?oN003ogT00oooo00<00?l@
0?ooo`3oool00`3oool00`00omh0oooo0?ooo`0=0?ooo`04003ogP3oool0oooo003ogQ@0oooo00D0
0:[o0?ooo`00omh0oooo003ogP040?ooo`0309co003oool0oooo00<0oooo00<00?oN0?ooo`3oool0
1`3oool00`00omh0oooo0000000=0?ooo`030000003oool0oooo0040oooo000I0?ooo`800000?`3o
ool00`00oj`0oooo0?ooo`060?ooo`03003ogP3oool00?oN00@0oooo00<00?oV0000003oool0@@3o
ool00`00oa00oooo0?ooo`0>0?ooo`03003ogP00onH0oooo0080oooo00<00?oV0?ooo`3oool01`3o
ool00`00onH0oooo0?ooo`090?ooo`05002^o`3oool00:ko0?ooo`00onH01P3oool00`2Do`00oooo
0?ooo`030?ooo`03003ogP3oool0oooo00D0oooo00<00?oN0?ooo`00WOl00P3oool00`000000oooo
003oiP0:0?ooo`030000003oool0oooo0040oooo000K0?ooo`030000003oool0oooo02`0oooo00<0
0?oN0?ooo`3oool03@3oool00`00OOl0oooo0?ooo`080?ooo`03003oiP3oool0oooo00<0oooo00<0
00000?ooo`3oool00`3oool00`00omh0oooo003ogP1;0?ooo`03003ogP00^_l0oooo00L0oooo00<0
0?oV0?ooo`3oool00P3oool00`00omh0oooo0?ooo`090?ooo`03002Zo`3oool0Y?l001H0oooo00<0
0?oN0?ooo`3=o`000`3oool00`0000000?oN003ogP090?ooo`030000003oool0oooo0040oooo000L
0?ooo`8000005`3oool00`00oh/0oooo0?ooo`030?ooo`03003ogP3oool00?oN00X0oooo00<00?oN
0?ooo`00omh03`3oool01000POl0oooo0?ooo`00okd=0?ooo`030000003oool0oooo00<0oooo00<0
0?oN0?ooo`00omh0A03oool01@00onH0oooo0?ooo`3oool00;ko00<0oooo00<00;ko0?ooo`3oool0
1P3oool00`00omh0oooo0?ooo`030?ooo`03003ogP3oool0oooo00P0oooo00<0[?l00?ooo`2/o`00
0`3oool01000onH0oooo0?ooo`00omh50?ooo`03003ogP3oool00?oN00T0oooo00L0cOl00?ooo`3o
ool0oooo0000003oool00?oN00X0oooo0@0000010?ooo`40oooo000N0?ooo`030000003oool0oooo
0140oooo00@00?n;0?ooo`3oool00?nD1@3oool00`00omh0oooo003ogP040?ooo`03003ogP3oool0
oooo00<0oooo00<00?oN0?ooo`3oool04P3oool00`00okd0oooo0?ooo`060?ooo`03003ogP3oool0
oooo0080oooo00<00000003oe@3oool0BP3oool01@00omh0oooo0?ooo`3oool00;[o00<0oooo00@0
J_l00?ooo`3oool00?oV1`3oool01@00onH0oooo0?ooo`3oool00?oV00X0oooo00<0Y?l00?ooo`3o
ool00`3oool01000omh0oooo0?ooo`00omh50?ooo`03002Eo`3oool00?oN0080oooo00<00?oN0?oo
o`3oool01`3oool00`00TOl0oooo0000000;0?ooo`4000000@3oool10?ooo`007`3oool00`000000
oooo0?ooo`0:0?ooo`03003oU03oool0oooo00<0oooo00@0023o0?ooo`3oool001co2P3oool00`00
omh0oooo003ogP0J0?ooo`03002jo`3oool0oooo00H0oooo00<00?oN0?ooo`3oool00P3oool00`00
00000?oN0?ooo`030?ooo`04003ogP3oool00?oN003oiT00oooo00<00;[o0?ooo`3oool0103oool0
1@1ko`00oooo0?ooo`3oool0Lol00080oooo00<00?oN003oiP3oool00`3oool00`00omh0oooo003o
gP0M0?ooo`030>ko003oool0oooo0080oooo00<00?oV003ogP3oool01`3oool0103oo`00oooo0?oo
o`00000:0?ooo`4000000@3oool10?ooo`00803oool2000000X0oooo00<00?nD0?ooo`3oool00`3o
ool010008?l0oooo0?ooo`007?l80?ooo`03003ogP3oool00?oN01`0oooo00@00;[o0?ooo`3oool0
0?o=2P3oool00`000000oooo0?ooo`030?ooo`04003ogP3oool00?oN003ogT00oooo00<00;[o0?oo
o`3oool0103oool00`1co`00oooo0?ooo`050?ooo`03002Mo`3oool0oooo0080oooo00<00?oN0?oo
o`3oool04P3oool00`00SOl0oooo002=o`020?ooo`03003ogP3oool00?oN00<0oooo00<0k_l00?oo
o`3oool00`3oool00`00omh0oooo0?ooo`060?ooo`050?Ko003oool0oooo0?ooo`0000002@3oool1
00000040oooo0@3oool00280oooo00<000000?ooo`3oool01`3oool00`00;?l0oooo0?ooo`090?oo
o`03003ogP3oool0oooo00<0oooo00<00?oN0?ooo`3oool00`3oool00`00omh0oooo0?ooo`0:0?oo
o`03003ogP3oool0oooo00`0oooo00<00?;o0?ooo`3oool0203oool00`0000000?oN0?ooo`170?oo
o`0307_o003oool0oooo00P0oooo00D00:7o0?ooo`3oool0oooo002Qo`0I0?ooo`0:002Ao`3oool0
097o0?oo003oool00?oN0?ooo`00omh0oooo003ogP@0oooo00<008Wo0?ooo`3oool02`3oool01000
QOl0oooo0?ooo`0000080?ooo`4000000@3oool10?ooo`008`3oool2000000L0oooo00<002So0?oo
o`3oool02@3oool00`00omh0oooo0?ooo`060?ooo`04003ogP3oool0oooo003ogPX0oooo00<00?oN
0?ooo`00omh03P3oool00`00l_l0oooo0?ooo`080?ooo`0300000000omD0oooo04L0oooo00<0Nol0
0?ooo`3oool01@3oool01000omh0oooo0?ooo`00WOl30?ooo`030<Go003oool0oooo00D0oooo00<0
0?oN0?ooo`3oool03`3oool0103onP00oooo0?ooo`3oo`020?ooo`04003ogP3oool0oooo003ogP@0
oooo00<0on800?ooo`3oool02`3oool01000QOl0077o0?ooo`0000020?ooo`03003oe@3oool0oooo
00<0oooo0@0000010?ooo`40oooo000U0?ooo`030000003oool0oooo0180oooo00@00?oN0?ooo`3o
ool00?nd0`3oool00`00omh0oooo0?ooo`050?ooo`03003ogP3oool0oooo00<0oooo00<00?oN0?oo
o`3oool06@3oool00`000000oooo0?ooo`0;0?ooo`03003ogP3oool0oooo03d0oooo00D009go0?oo
o`3oool0oooo003ogP020?ooo`050=Go003oool0oooo0?ooo`3=o`000P3oool00`00omh0oooo0?oo
o`020?ooo`03003oiP3oool0oooo00H0oooo00<00?oN0?ooo`3oool01P3oool01`3oo`00oooo0?oo
o`00OOl0oooo0021o`00onH01`3oool00`3oiP00oooo0?ooo`0<0?ooo`060?oN003oool0oooo0000
003oool00?oN1@3oool100000040oooo0@3oool002H0oooo0P00000B0?ooo`04003o[000omh0oooo
002Eo`/0oooo00<00?oN0?ooo`3oool07`3oool00`000000oooo003ogP0;0?ooo`03003ogP3oool0
oooo03d0oooo00<009go0?ooo`3oool0103oool00`3=o`00oooo0?ooo`040?ooo`03003ogP3oool0
oooo00H0oooo00<00?oE0?ooo`3oool00P3oool00`00omh0oooo0?ooo`090?ooo`030021o`3oool0
olh000X0oooo00<007Wo0?ooo`3oool02P3oool01@3ogP00oiT00?ooo`3oool0000000L0oooo0@00
00010?ooo`00:03oool00`000000oooo0?ooo`090?ooo`03003o]03oool0oooo00<0oooo00@009Go
003ogP3oool0097o0P3oool00`00olD0oooo0?ooo`0S0?ooo`03003omP3oool0oooo0080oooo00<0
00000?ooo`00omh0B`3oool00`3Eo`00oooo0?ooo`040?ooo`03003oiP3oool0oooo00H0oooo00<0
087o003ogP3oool00P3oool00`00onH0oooo003ogP0@0?ooo`030?o:003oool0olh000X0oooo00<0
okh00?ooo`00JOl00`3oool00`00omh0oooo0?ooo`050?ooo`050?nI003oool0oooo0?ooo`000000
1P3oool100000040oooo000Y0?ooo`030000003oool0oooo00P0oooo00<00?nd0?ooo`3oool00`3o
ool01000UOl0oooo0?ooo`00olD20?ooo`03003o_@3oool0oooo00L0oooo00<00?oN0?ooo`3oool0
6@3oool00`00oi@0oooo0?ooo`020?ooo`030000003oool0oooo04/0oooo00<0eOl00?ooo`3oool0
103oool00`00omh0oooo0?ooo`020?ooo`060021o`3oool0oooo0?ooo`00QOl00?oN0`3oool00`00
omh0oooo0?ooo`0@0?ooo`060?o:003oool0oooo001ao`3oool007Go1`3oool00`3o^P00oooo001U
o`030?ooo`03003oe@3oool0oooo00X0oooo00<000000?ooo`3oool00`3oool100000040oooo000Z
0?ooo`800000203oool01@00WOl0oooo0?ooo`3oool00?o500@0oooo00@00?o50?ooo`3oool00<Ko
0P3oool01@00omh0oooo0?ooo`3oool00?oN0080oooo00<00?oN0?ooo`3oool02@3oool00`00onH0
oooo003oiP0=0?ooo`03003oW03oool0oooo0080oooo00<000000?ooo`3oool0E`3oool01P00POl0
oooo0?ooo`3oool0omH0003oiPD0oooo00<00?oV0?ooo`3oool04@3oool01000MOl0oooo0?nU0000
GOl40?ooo`05003ogP3oool0oooo0?ooo`3oM@00403oool00`000000oooo0?ooo`030?ooo`400000
0@3oool002`0oooo00<000000?ooo`3oool01@3oool01@00VOl0oooo0?ooo`3oool00?o500@0oooo
00L00<[o0?ooo`3oool00<Ko003oe@00omh00?oE00<0oooo00<00?oN0?ooo`3oool0303oool00`00
omh0oooo003ogP0B0?ooo`030000003oool0oooo0100oooo00<00?oN0?ooo`3oool0?@3oool00`00
OOl0oooo0?ooo`020?ooo`03003ogP3oool0olh000<0oooo00@0omH00?ooo`00omh0077o0`3oool0
0`00omh0oooo0?ooo`0A0?ooo`040?nM0000GOl0oj40001Mo`80oooo00<00?oE0?ooo`00omD00`3o
ool00`3oL@00oooo0?ooo`0?0?ooo`030000003oool0oooo0080oooo0@0000010?ooo`00;@3oool2
000000T0oooo00<00<ko0?ooo`3oool00P3oool01`00a_l0oooo0?ooo`3oool00?o=003ogP00m_l0
303oool00`00onH0oooo0?ooo`0H0?ooo`030000003oool0oooo00H0oooo00<00?oV0?ooo`3oool0
1`3oool00`00onH0oooo0?ooo`0m0?ooo`030021o`3oool0oooo0080oooo00<00?oN0?ooo`3ocP00
0P3oool01@00LOl0oooo0?ooo`00omh007Go01L0oooo00@0oj400?m@003oool0oeT00P3oool00`00
omh0oooo0?ooo`0D0?ooo`03003oL`3oool0000000<0oooo0@0000010?ooo`00;`3oool00`000000
oooo0?ooo`060?ooo`04003:o`3oool0oooo003oe@@0oooo00@00?Ko0?ooo`3oool00?[o303oool0
0`00omh0oooo0?ooo`0:0?ooo`03003ogP3oool0oooo00/0oooo00<00000003Vo`3oool01P3oool0
0`00omh0oooo0?ooo`170?ooo`030?o6003oool0oooo00@0oooo00@00?oN0?ooo`3oool0077o0`3o
ool00`3oW@00oooo0?ooo`0F0?ooo`030?mE003oool0oeT001T0oooo00@0@@3o0?ooo`3oool00000
0P3oool100000040oooo000`0?ooo`030000003oool0oooo00P0oooo00<00?[o0?ooo`3oool00P3o
ool00`00n_l0oooo0?ooo`0=0?ooo`03003ogP3oool0oooo00P0oooo00<00?oN0?ooo`00omh03@3o
ool00`0000000?lH0?ooo`0D0?ooo`03003oiP3oool0oooo03T0oooo00@0olH00?ooo`3oool006go
1P3oool01@3oV@00oooo0?ooo`3oool0oj400080oooo00<006Go0?ooo`3oool08`3oool00`00ofX0
oooo0?ooo`090?ooo`070440o`3oool0oooo0?ooo`000000oooo000000010?ooo`00<@3oool20000
00P0oooo00<00?[o0?ooo`3oool02`3oool00`00onh0oooo0?ooo`020?ooo`03003ogP3oool00?oN
00X0oooo00<00?oN0?ooo`3oool0303oool00`000000oooo003o600E0?ooo`03003ogP3oool0oooo
03`0oooo00<006go0?ooo`3oool0103oool00`3oU@00oooo001Qo`040?ooo`03001Uo`3oool0oooo
02<0oooo00<00?mZ0?ooo`3oool03P3oool100000040oooo0@00000003<0oooo00<000000?ooo`3o
ool04@3oool00`00onh00?oN003okP040?ooo`03003ogP3oool0oooo00@0oooo00<00?oN0?ooo`3o
ool04P3oool00`000000oooo0?ooo`0;0?ooo`03003oiP3oool0oooo00X0oooo00<00?oN0?ooo`3o
ool0>@3oool00`3oS@00oooo0?ooo`060?ooo`03001Uo`3oool0oooo0080oooo00<0og40001<o`3o
ool00`3oool00`00omh0oooo0?ooo`0E0?ooo`03003oHP3oool0oooo00D0oooo00<0FP3o0?ooo`3o
ool03P3oool100000040oooo0@00000003@0oooo0P00000;0?ooo`03003okP3oool0oooo00<0oooo
00<00?nT003ogP00oj@02`3oool00`00omh0oooo0?ooo`0B0?ooo`030000003oool0oooo00/0oooo
00<00?oN0?ooo`3oool02P3oool00`00onH0oooo0?ooo`0i0?ooo`040?n=003oool0oooo001Mo`H0
oooo00L0ofT00018o`3oool0oooo0?ma003o70000?oN00<0oooo00<00?oN0?ooo`3oool04P3oool0
1000of80oooo0?ooo`00of870?ooo`0305X0o`3oool0oooo00l0oooo0P00000003H0oooo00<00000
0?ooo`3oool0203oool00`00onh0oooo0?ooo`030?ooo`03003oY03oool00?nL01T0oooo00<00?oN
0?ooo`3oool0103oool00`000000oooo0?ooo`070?ooo`03002no`3oool0oooo00l0oooo00<00?oV
0?ooo`3oool0>`3oool00`00HOl004Co0?ooo`030?ooo`03003ogP3oJ@00oaP000<0oooo00<0ob00
003ogP3oool05`3oool01000ofX0JP3o0?ooo`1Z0?lH0?ooo`<00000000g0?ooo`800000203oool0
0`00oj@0oooo0?ooo`0M0?ooo`03003ogP3oool00?oN00H0oooo00<000000?ooo`3oool01`3oool0
0`00^_l0oooo0?ooo`0?0?ooo`03003ogP3oool0oooo03/0oooo00<0oed00014o`3oool00`3oool0
0`00omD0oooo0?lD000N0?ooo`0307<0o`3oool0JP3o01@0oooo100000030?ooo`00>@3oool00`00
0000oooo0?ooo`050?ooo`03003oY03oool0oooo00`0oooo00<00>ko0?ooo`3oool00P3oool01000
omh0oooo0?ooo`00onH20?ooo`03003ogP3oool0oooo00<0oooo00<00?oN0?ooo`3oool01P3oool0
0`000000oooo0?ooo`060?ooo`0306[o003oool0oooo04h0oooo00<0of400?l8003oool0=P3oool5
000000L0oooo000j0?ooo`030000003oool0oooo0140oooo00<00>ko0?ooo`00j_l0103oool01000
omh0oooo0?ooo`00omh20?ooo`03003ogP3oool0oooo00H0oooo00<00?oN0?ooo`00omh00`3oool0
0`000000oooo0?ooo`060?ooo`0307?o003oool0oooo00P0oooo00<00?oN0?ooo`3oool0A03oool0
0`3o2000oooo0?ooo`0a0?ooo`@00000303oool003/0oooo0P00000;0?ooo`03003bo`3oool0oooo
0080oooo00@00?la003^o`3oool00?la4`3oool00`00onH0oooo003oiP030?ooo`030000003oool0
oooo00/0oooo00<00:7o0?ooo`3oool00`3oool00`00onH0oooo0?ooo`1E0?ooo`03003oFP3oool0
oooo01`0oooo1000000@0?ooo`00?@3oool00`000000oooo0?ooo`080?ooo`03003^o`3oool0oooo
0080oooo00@00?la0?ooo`3oool00?la303oool00`00omh0oooo003ogP0:0?ooo`03003ogP3oool0
oooo00/0oooo00<00:7o0?ooo`3oool04P3oool00`00omh0oooo0?ooo`120?ooo`05003oDP3oool0
oooo0?ooo`2;0?l06@3oool5000001@0oooo000n0?ooo`800000203oool00`00ocT0oooo0?ooo`0?
0?ooo`06003oiP3oool0oooo003ogP3oool00?oN1`3oool01@00omh0oooo0?ooo`00onH00?oN00`0
oooo00<0cOl00?ooo`3oool01P3oool00`00omh0oooo0?ooo`090?ooo`03003ogP3oool0oooo03/0
oooo00<00?mB0?ooo`3oool0103oool01@2L0?l0oooo0?ooo`3oool0U03o01D0oooo1000000I0?oo
o`00@03oool00`000000oooo0?ooo`050?ooo`03003o>@3oool0oooo00X0oooo00<00?oN0?ooo`3o
ool00P3oool00`00omh0oooo0?ooo`050?ooo`03003ogP3oool0oooo0080oooo00D00?oN0?ooo`00
onH00000003oiP0<0?ooo`030<Go003oool0oooo00H0oooo00<00?oN0?ooo`3oool0A`3oool00`00
oeX0oooo0?ooo`040?ooo`0309@0o`3oool0oooo0180oooo1@00000M0?ooo`00@@3oool00`000000
oooo0?ooo`0A0?ooo`03003:o`3oool00<[o00X0oooo00<00?oV0?ooo`00onH0103oool00`00omh0
00000?ooo`0I0?ooo`03003ogP3oool0oooo04<0oooo00<0W03o0?ooo`3oool05@3oool400000280
oooo00120?ooo`8000004@3oool00`00b_l0oooo047o000<0?ooo`03003ogP3oool0oooo00<0oooo
00<000000?ooo`3oool04@3oool00`00POl0oooo0?ooo`040?ooo`03003ogP3oool0oooo04<0oooo
00<0W03o0?ooo`3oool04@3oool4000002H0oooo00140?ooo`030000003oool0oooo00P0oooo00<0
0<ko0?ooo`3oool00`3oool00`11o`00oooo047o00080?ooo`03003oiP3oool00?oV00L0oooo00<0
00000?ooo`3oool04@3oool00`3ofP00oooo0?ooo`050?ooo`03003oiP3oool0oooo0540oooo1@00
000Z0?ooo`00A@3oool2000000P0oooo00<0>Ol00?ooo`3oool00`3oool00`11o`00oooo0?ooo`04
0?ooo`07002bo`3oool0oooo0?ooo`00omh0oooo003ogP070?ooo`030000003oool0oooo0140oooo
00<0omH00?ooo`3oool01@3oool00`00omh0oooo0?ooo`1=0?ooo`@00000;`3oool004L0oooo00<0
00000?ooo`3oool01@3oool00`0ao`00oooo0?ooo`080?ooo`03002fo`00omh00;;o00T0oooo00D0
0?oV0?ooo`3oool0oooo000000050?ooo`03003ogP3oool00?oN00h0oooo00<007Go0?ooo`3oool0
C@3oool5000003<0oooo00180?ooo`8000002P3oool00`00]_l0oooo0?ooo`030?ooo`0308_o0000
omh0Rol000L0oooo00<00?oN0?ooo`00omh00`3oool00`000000oooo0?ooo`030?ooo`03003oe@3o
ool00?oE00h0oooo00<0oj400?ooo`3oool0B@3oool4000003P0oooo001:0?ooo`030000003oool0
oooo00L0oooo00<00;[o0?ooo`3oool00`3oool01@2;o`00oooo09Co003oool00?oV00D0oooo00<0
0?oV0?ooo`3oool00`3oool00`0000000?oN0?ooo`0D0?ooo`050?nQ003oool0oooo0?ooo`00IOl0
@P3oool5000003`0oooo001;0?ooo`030000003oool0oooo00H0oooo00<0Nol00?ooo`3oool01`3o
ool00`00omh0oooo0?ooo`080?ooo`03003ogP0000000?oN01T0oooo00<006Wo001<o`3oool01@3o
ool00`00omh0oooo0?ooo`0d0?ooo`@00000@@3oool004`0oooo0P0000060?ooo`0308?o003oool0
oooo00T0oooo00<009Wo0?ooo`00VOl0103oool01@00omh0oooo003ogP0000000?oN01T0oooo00<0
ogD0001@o`3oool01@3oool00`00omh0oooo0?ooo`0`0?ooo`@00000A@3oool004h0oooo00<00000
0?ooo`3oool03`3oool00`00VOl0oooo0>Ko00040?ooo`05003ogP3oool00?oN00000000omh06@3o
ool00`3oM@00ob000?ooo`0c0?ooo`D00000B@3oool004l0oooo0P0000090?ooo`03002Qo`3oool0
oooo00<0oooo00@0g_l00?ooo`3Vo`000?oN0P3oool01@00ROl0oooo0?ooo`00omh0000001/0oooo
00<0ob@00?ooo`3oool0;P3oool4000004h0oooo001A0?ooo`030000003oool0oooo00H0oooo00<0
cOl00?ooo`3oool00`3oool0103No`00oooo0?ooo`00SOl20?ooo`050029o`3oool0oooo0?ooo`00
0000A`3oool500000580oooo001B0?ooo`8000001P3oool01@3=o`00oooo0?ooo`3oool0097o00@0
oooo00@0oo800?ooo`3oool0onh00`3oool00`000000oooo0?ooo`110?ooo`@00000E`3oool005@0
oooo00<000000?ooo`3oool01`3oool00`00TOl0oooo0?ooo`020?ooo`080?ob003oool0oooo0?oZ
0000POl0oooo001mo`0000070?ooo`03003oe@3oool0oooo01P0oooo00<00?mJ0?ooo`3oool06P3o
ool4000005/0oooo001E0?ooo`030000003oool0oooo00H0oooo00<0ool00?ooo`3oool01P3oool0
1@00POl00?oV0?o20000000006Wo00@0oooo00<00?oN0?ooo`00omh06P3oool00`2;0?l0oooo0?oo
o`0E0?ooo`D00000G`3oool005H0oooo0P0000060?ooo`040?Ko003oool0oooo0025o`D0oooo00D0
olX0001Yo`3o`P000?oE0?mm00040?ooo`03003oe@3oool0oooo01X0oooo00<0R`3o0?ooo`3oool0
4@3oool4000006@0oooo001H0?ooo`030000003oool0oooo00H0oooo00<0omh00?ooo`00LOl00`3o
ool01@3obP00ohD000000000omh0ogT00300oooo1@00001X0?ooo`00F@3oool2000000H0oooo00<0
omh00?ooo`3oV@00103oool00`3oP@0000000?ooo`0]0?ooo`@00000K@3oool005/0oooo00<00000
0?ooo`3oool01@3oool00`3oV@00oooo0?ooo`030?ooo`030000003oool0oooo02L0oooo1@00001a
0?ooo`00G03oool00`000000oooo0?ooo`0:0?ooo`030000003oool0oooo00H0oooo00<00?mZ0?oo
o`3oool06P3oool4000007H0oooo001M0?ooo`8000002P3oool00`000000oooo0?ooo`030?ooo`04
003oJP3oool0oooo003oJQP0oooo1000001j0?ooo`00G`3oool00`000000oooo0?ooo`070?ooo`03
00000000ofX0oooo00<0oooo00@0DP3o0?ooo`3oool0DP3o4`3oool5000007h0oooo001P0?ooo`80
00001`3oool00`000000BP3o0?ooo`030?ooo`040580o`3oool0oooo05X0o`l0oooo100000230?oo
o`00HP3oool00`000000oooo0?ooo`040?ooo`03000000110?l0oooo0140oooo1@0000270?ooo`00
H`3oool2000000@0oooo00<000000?ooo`3oool03@3oool4000008`0oooo001U0?ooo`050000003o
ool0oooo0?ooo`0000002`3oool400000900oooo001V0?ooo`040000003oool0oooo000000H0oooo
1@00002D0?ooo`00I`3oool300000080oooo1000002I0?ooo`00J@3oool3000009d0oooo0000\
\>"],
  ImageRangeCache->{{{0, 264.312}, {286.938, 0}} -> {0.0486402, -4.72903, \
0.00324458, 0.00324458}}]
}, Open  ]],

Cell[TextData[{
  "For a moderate sized tensor product grid like the example above, using \
Apply is reasonably fast. However, as the grid size gets larger, this \
approach may not be the fastest because Apply can only be used in limited \
ways with the ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " compiler and hence, with packed arrays.",
  " ",
  "If you can define your function so you can use ",
  StyleBox["Map", "MR"],
  " instead of ",
  StyleBox["Apply", "MR"],
  ", you may be able to use a ",
  StyleBox["CompiledFunction", "MR"],
  " since ",
  StyleBox["Map", "MR"],
  " has greater applicability within the ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " compiler than does ",
  StyleBox["Apply", "MR"],
  "."
}], "Text"],

Cell[TextData[{
  "This defines a ",
  StyleBox["CompiledFunction", "MR"],
  " which using ",
  StyleBox["Map", "MR"],
  " to get the values on the grid.",
  " ",
  "(If the first grid dimension is greater than the SystemOption \"",
  StyleBox["MapCompileLength", "MR"],
  "\", then you do not need to construct the ",
  StyleBox["CompiledFunction", "MR"],
  " since the compilation is done automatically when grid is a packed \
array.)"
}], "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(cf\  = \ 
      Compile[{{grid, \ _Real, \ 4}}, \ 
        Map[Function[{X}, \ 
            Module[{Xs\  = \ X\  - \  .5}, \ Exp[\(-\((Xs . Xs)\)\)]]], 
          grid, \ {3}]]\)], "Input",
  CellLabel->"In[49]:="],

Cell[BoxData[
    TagBox[\(CompiledFunction[{grid}, 
        Map[Function[{X}, 
            Module[{Xs = X - 0.5`}, \[ExponentialE]\^\(-Xs . Xs\)]], 
          grid, {3}], "-CompiledCode-"]\),
      False,
      Editable->False]], "Output",
  CellLabel->"Out[49]="]
}, Open  ]],

Cell[TextData[{
  "An even better approach, when possible, is to take advantage of \
listability when your function consists of operations and functions which \
have the Listable attribute. The trick is to separate the x, y, and z values \
at each of the points on the tensor product grid. The fastest way to do this \
is using ",
  StyleBox["Transpose[fddf@Grid[], RotateLeft[Range[n + 1]]]", "MR"],
  ", where n = ",
  StyleBox["Length[fddf@DerivativeOrder[]]", "MR"],
  " is the dimensionality of the space in which you are approximating the \
derivative. This will return a list of length n, which has the values on the \
grid for each of the component dimensions separately.",
  " ",
  "With the ",
  StyleBox["Listable", "MR"],
  " attribute, functions applied to this will thread over the grid."
}], "Text"],

Cell["\<\
This defines a function that takes advantage of the fact that Exp \
has the Listable attribute to find the values on the grid.\
\>", "MathCaption"],

Cell[BoxData[
    \(fgrid[grid_]\  := \ 
      Apply[f, \ 
        Transpose[grid, \ 
          RotateLeft[Range[TensorRank[grid]], \ 1]]]\)], "Input",
  CellLabel->"In[50]:="],

Cell["\<\
This compares timings for the three methods. The commands are \
repeated several times to get more accurate timings.\
\>", "MathCaption"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Module[{repeats\  = \ 100, grid\  = \ fddf@Grid[], \ 
        n\  = \ Length[fddf@DerivativeOrder[]]}, \[IndentingNewLine]{First[
          Timing[Do[
              Apply[f, \ grid, \ {n}], {repeats}]]], \[IndentingNewLine]First[
          Timing[Do[cf[grid], {repeats}]]], \[IndentingNewLine]First[
          Timing[Do[fgrid[grid], {repeats}]]]}]\)], "Input",
  CellLabel->"In[51]:="],

Cell[BoxData[
    \({9.514`\ Second, 0.7609999999999992`\ Second, 
      0.5710000000000015`\ Second}\)], "Output",
  CellLabel->"Out[51]="]
}, Open  ]],

Cell[TextData[{
  "The example timings show that using the ",
  StyleBox["CompiledFunction", "MR"],
  " is typically much faster than using ",
  StyleBox["Apply", "MR"],
  " and taking advantage of listability is a little faster yet."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Pseudospectral Derivatives", "Subsubsection",
  CellTags->{"c:5", "b:7.1.2", "ndsg:2.0.1.2", "PDE:1.2.4"}],

Cell["\<\
The maximum value the difference order can take on is determined by \
the number of points in the grid. If you exceed this, a warning message will \
be given and the order reduced automatically.\
\>", "Text",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell["\<\
This uses maximal order to approximate the first derivative of the \
sine function on a random grid. \
\>", "MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(NDSolve`FiniteDifferenceDerivative[1, \ rgrid, \ Sin[rgrid], 
      DifferenceOrder \[Rule] Length[rgrid]]\)], "Input",
  CellLabel->"In[52]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[BoxData[
    \(NDSolve`FiniteDifferenceDerivative::"ordred" \(\(:\)\(\ \)\) 
      "There are insufficient points in dimension \!\(1\) to achieve the \
requested approximation order. Order will be reduced to \!\(11\)."\)], \
"Message",
  CellLabel->"From In[52]:="],

Cell[BoxData[
    \({0.9994144086182475`, 
      0.27201964459042566`, \(-0.8787660321914593`\), \
\(-0.9322198199141845`\), \(-0.9754338570865738`\), \(-0.9846801715860887`\), \
\(-0.5203816246318177`\), \(-0.16972003855257667`\), 0.28697067064065324`, 
      0.5056926432204779`, 0.5879469309963493`, 
      1.000021962596272`}\)], "Output",
  CellLabel->"Out[52]="]
}, Open  ]],

Cell[TextData[{
  "Using a limiting order is commonly referred to as a ",
  StyleBox["pseudospectral",
    FontWeight->"Bold"],
  " derivative. A common problem with these is that artificial oscillations \
(Runge's phenomena) can be extreme. However, there are two instances where \
this is not the case: a uniform grid with periodic repetition and a grid with \
points at the zeros of the Chebyshev polynomials, ",
  Cell[BoxData[
      \(TraditionalForm\`T\_n\)]],
  ", or Chebyshev-Gauss-Lobatto points [",
  ButtonBox["F96",
    ButtonData:>"F96",
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["QV94",
    ButtonData:>"QV94",
    ButtonStyle->"Hyperlink"],
  "].",
  " ",
  "The computation in both of these cases can be done using a fast Fourier \
transform, which is efficient and minimizes roundoff error."
}], "Text",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[BoxData[
    FormBox[GridBox[{
          {
            RowBox[{
              StyleBox["DifferenceOrder",
                FontFamily->"Courier"], 
              StyleBox["->",
                FontFamily->"Courier"], 
              StyleBox["n",
                "TI"]}], Cell[TextData[Cell[BoxData[
                \(TraditionalForm\`use\ \(n\^th\) 
                  order\ finite\ differences\ to\ approximate\ the\ \
derivative\)]]], "Text"]},
          {
            RowBox[{
              StyleBox["DifferenceOrder",
                FontFamily->"Courier"], 
              StyleBox["->",
                FontFamily->"Courier"], 
              RowBox[{
                StyleBox["Length",
                  FontFamily->"Courier"], 
                StyleBox["[",
                  FontFamily->"Courier"], 
                StyleBox["grid",
                  FontFamily->"Courier",
                  FontSlant->"Italic"], 
                StyleBox["]",
                  "MR"]}]}], Cell[TextData[{
              
              "use the highest possible order finite differences to \
approximate derivative on ",
              StyleBox["grid", "MR",
                FontSlant->"Italic"],
              "  (Not generally recommended.)"
            }], "Text"]},
          {Cell[TextData[StyleBox[
            "DifferenceOrder->\"Pseudospectral\"", "MR"]]], Cell[TextData[{
              
              "use a pseudospectral derivative approximation.  Only \
applicable when the grid points are spaced corresponding to the \
Chebyshev-Gauss-Lobatto points or when the grid is uniform with ",
              StyleBox["PeriodicInterpolation->True", "MR"],
              "."
            }], "Text"]},
          {
            StyleBox[
              RowBox[{"DifferenceOrder", "\[Rule]", 
                RowBox[{"{", 
                  RowBox[{
                    StyleBox["n1",
                      FontSlant->"Italic"], ",", " ", 
                    StyleBox["n2",
                      FontSlant->"Italic"], 
                    StyleBox[",",
                      FontSlant->"Italic"], 
                    StyleBox[" ",
                      FontSlant->"Italic"], 
                    StyleBox["...",
                      FontSlant->"Italic"]}], "}"}]}],
              FontFamily->"Courier"], Cell[TextData[{
              "use difference orders ",
              StyleBox["n1, n2", "MR",
                FontSlant->"Italic"],
              ", ... in dimensions 1, 2, ... respectively"
            }], "Text"]}
          }], TextForm]], "DefinitionBox",
  GridBoxOptions->{ColumnWidths->{0.45, 0.55}}],

Cell["Settings for the DifferenceOrder option. ", "Caption"],

Cell["\<\
This gives a pseudospectral approximation for the second derivative \
of the sine function on a uniform grid.\
\>", "MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(ugrid\  = \ 
        N[2\ \[Pi]\ Range[0, 10]/10];\)\), "\[IndentingNewLine]", 
    \(NDSolve`FiniteDifferenceDerivative[1, \ ugrid, \ Sin[ugrid], \ 
      PeriodicInterpolation \[Rule] True, \ 
      DifferenceOrder -> "\<Pseudospectral\>"]\)}], "Input",
  CellLabel->"In[53]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[BoxData[
    \({1.0000000000000002`, 0.8090169943749471`, 
      0.30901699437494756`, \(-0.30901699437494734`\), \(-0.8090169943749473`\
\), \(-0.9999999999999998`\), \(-0.8090169943749473`\), \
\(-0.30901699437494745`\), 0.309016994374947`, 0.8090169943749476`, 
      1.0000000000000002`}\)], "Output",
  CellLabel->"Out[54]="]
}, Open  ]],

Cell["\<\
This computes the error at each point. The approximation is \
accurate to roundoff because the effective basis for the pseudospectral \
derivative on a uniform grid for a periodic function are the trigonometric \
functions.\
\>", "MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(%\  - \ Cos[ugrid]\)], "Input",
  CellLabel->"In[55]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[BoxData[
    \({2.220446049250313`*^-16, \(-3.3306690738754696`*^-16\), 
      1.1102230246251565`*^-16, 0.`, 0.`, 2.220446049250313`*^-16, 
      2.220446049250313`*^-16, 
      1.1102230246251565`*^-16, \(-2.220446049250313`*^-16\), 
      2.220446049250313`*^-16, 2.220446049250313`*^-16}\)], "Output",
  CellLabel->"Out[55]="]
}, Open  ]],

Cell[TextData[{
  "The Chebyshev-Gauss-Lobatto points are the zeros of ",
  Cell[BoxData[
      \(TraditionalForm\`\((1\  - \ x\^2)\) \(\(T\_n\^'\)(x)\)\)]],
  ".  Using the property ",
  Cell[BoxData[
      \(TraditionalForm\`\(T\_n\)(x)\  = \ \(T\_n\)(cos(\[Theta])) == 
          cos(n\ \[Theta])\)]],
  ", these can be shown to be at ",
  Cell[BoxData[
      \(TraditionalForm\`x\_j = cos(\(\[Pi]\ j\)\/n)\)]],
  " ."
}], "Text",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell["\<\
This defines a simple function that generates a grid of n points \
with leftmost point at x0 and interval length L having the spacing of the \
Chebyshev-Gauss-Lobatto points\
\>", "MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[BoxData[
    \(CGLGrid[x0_, \ L_, \ n_Integer\  /; \ n\  > \ 1]\  := \ 
      x0\  + 1\/2\ L \((1\  - \ \ Cos[\[Pi]\ Range[0, 
                      n - 1]/\((n - 1)\)])\)\)], "Input",
  CellLabel->"In[56]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell["\<\
This computes the pseudospectral derivative for a Gaussian \
function.\
\>", "MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(cgrid\  = \ CGLGrid[\(-5\), 10. , 16]; 
    NDSolve`FiniteDifferenceDerivative[1, cgrid, \ Exp[\(-cgrid\^2\)], 
      DifferenceOrder -> "\<Pseudospectral\>"]\)], "Input",
  CellLabel->"In[57]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[BoxData[
    \({0.04024259748482765`, \(-0.020992247826050765`\), 
      0.023915096104594188`, \(-0.0300589226076537`\), 
      0.04255532278960516`, \(-0.05908708518729287`\), 0.40662970477175964`, 
      0.6033596780293704`, \(-0.6033596780293702`\), \
\(-0.40662970477175964`\), 0.05908708518729286`, \(-0.042555322789605096`\), 
      0.03005892260765353`, \(-0.023915096104594195`\), 
      0.020992247826051247`, \(-0.040242597484827745`\)}\)], "Output",
  CellLabel->"Out[57]="]
}, Open  ]],

Cell["This shows a plot of the approximation and the exact values.", \
"MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Show[
        Block[{$DisplayFunction\  = \ 
              Identity}, \ {\[IndentingNewLine]ListPlot[
              Transpose[{cgrid, \ %}], \ 
              PlotStyle \[Rule] PointSize[0.025]], \[IndentingNewLine]Plot[
              Evaluate[D[Exp[\(-x\^2\)], x]], {x, \(-5\), 5}]}], \ 
        PlotRange \[Rule] All];\)\)], "Input",
  CellLabel->"In[58]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.0952381 0.30901 0.343123 [
[.11905 .29651 -6 -9 ]
[.11905 .29651 6 0 ]
[.30952 .29651 -6 -9 ]
[.30952 .29651 6 0 ]
[.69048 .29651 -3 -9 ]
[.69048 .29651 3 0 ]
[.88095 .29651 -3 -9 ]
[.88095 .29651 3 0 ]
[.4875 .05167 -30 -4.5 ]
[.4875 .05167 0 4.5 ]
[.4875 .13745 -24 -4.5 ]
[.4875 .13745 0 4.5 ]
[.4875 .22323 -30 -4.5 ]
[.4875 .22323 0 4.5 ]
[.4875 .39479 -24 -4.5 ]
[.4875 .39479 0 4.5 ]
[.4875 .48057 -18 -4.5 ]
[.4875 .48057 0 4.5 ]
[.4875 .56635 -24 -4.5 ]
[.4875 .56635 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.11905 .30901 m
.11905 .31526 L
s
[(-4)] .11905 .29651 0 1 Mshowa
.30952 .30901 m
.30952 .31526 L
s
[(-2)] .30952 .29651 0 1 Mshowa
.69048 .30901 m
.69048 .31526 L
s
[(2)] .69048 .29651 0 1 Mshowa
.88095 .30901 m
.88095 .31526 L
s
[(4)] .88095 .29651 0 1 Mshowa
.125 Mabswid
.16667 .30901 m
.16667 .31276 L
s
.21429 .30901 m
.21429 .31276 L
s
.2619 .30901 m
.2619 .31276 L
s
.35714 .30901 m
.35714 .31276 L
s
.40476 .30901 m
.40476 .31276 L
s
.45238 .30901 m
.45238 .31276 L
s
.54762 .30901 m
.54762 .31276 L
s
.59524 .30901 m
.59524 .31276 L
s
.64286 .30901 m
.64286 .31276 L
s
.7381 .30901 m
.7381 .31276 L
s
.78571 .30901 m
.78571 .31276 L
s
.83333 .30901 m
.83333 .31276 L
s
.07143 .30901 m
.07143 .31276 L
s
.02381 .30901 m
.02381 .31276 L
s
.92857 .30901 m
.92857 .31276 L
s
.97619 .30901 m
.97619 .31276 L
s
.25 Mabswid
0 .30901 m
1 .30901 L
s
.5 .05167 m
.50625 .05167 L
s
[(-0.75)] .4875 .05167 1 0 Mshowa
.5 .13745 m
.50625 .13745 L
s
[(-0.5)] .4875 .13745 1 0 Mshowa
.5 .22323 m
.50625 .22323 L
s
[(-0.25)] .4875 .22323 1 0 Mshowa
.5 .39479 m
.50625 .39479 L
s
[(0.25)] .4875 .39479 1 0 Mshowa
.5 .48057 m
.50625 .48057 L
s
[(0.5)] .4875 .48057 1 0 Mshowa
.5 .56635 m
.50625 .56635 L
s
[(0.75)] .4875 .56635 1 0 Mshowa
.125 Mabswid
.5 .06882 m
.50375 .06882 L
s
.5 .08598 m
.50375 .08598 L
s
.5 .10314 m
.50375 .10314 L
s
.5 .12029 m
.50375 .12029 L
s
.5 .1546 m
.50375 .1546 L
s
.5 .17176 m
.50375 .17176 L
s
.5 .18892 m
.50375 .18892 L
s
.5 .20607 m
.50375 .20607 L
s
.5 .24039 m
.50375 .24039 L
s
.5 .25754 m
.50375 .25754 L
s
.5 .2747 m
.50375 .2747 L
s
.5 .29185 m
.50375 .29185 L
s
.5 .32617 m
.50375 .32617 L
s
.5 .34332 m
.50375 .34332 L
s
.5 .36048 m
.50375 .36048 L
s
.5 .37763 m
.50375 .37763 L
s
.5 .41195 m
.50375 .41195 L
s
.5 .4291 m
.50375 .4291 L
s
.5 .44626 m
.50375 .44626 L
s
.5 .46342 m
.50375 .46342 L
s
.5 .49773 m
.50375 .49773 L
s
.5 .51488 m
.50375 .51488 L
s
.5 .53204 m
.50375 .53204 L
s
.5 .5492 m
.50375 .5492 L
s
.5 .03451 m
.50375 .03451 L
s
.5 .01736 m
.50375 .01736 L
s
.5 .0002 m
.50375 .0002 L
s
.5 .58351 m
.50375 .58351 L
s
.5 .60066 m
.50375 .60066 L
s
.5 .61782 m
.50375 .61782 L
s
.25 Mabswid
.5 0 m
.5 .61803 L
s
.025 w
.02381 .32282 Mdot
.03422 .30181 Mdot
.06498 .31722 Mdot
.11475 .2987 Mdot
.18137 .32361 Mdot
.2619 .28874 Mdot
.35285 .44853 Mdot
.45022 .51604 Mdot
.54978 .10198 Mdot
.64715 .16949 Mdot
.7381 .32928 Mdot
.81863 .29441 Mdot
.88525 .31932 Mdot
.93502 .3008 Mdot
.96578 .31621 Mdot
.97619 .2952 Mdot
.5 Mabswid
.02381 .30901 m
.02605 .30901 L
.02846 .30901 L
.03279 .30901 L
.03543 .30901 L
.03793 .30901 L
.04018 .30901 L
.04262 .30901 L
.045 .30901 L
.04762 .30901 L
.05036 .30901 L
.05293 .30901 L
.05751 .30901 L
.06009 .30901 L
.06244 .30901 L
.06504 .30901 L
.0678 .30901 L
.07041 .30901 L
.07277 .30901 L
.07556 .30901 L
.07858 .30901 L
.08143 .30901 L
.08405 .30901 L
.08881 .30901 L
.09151 .30901 L
.09403 .30901 L
.09912 .30901 L
.10198 .30901 L
.10458 .30901 L
.10939 .30901 L
.11181 .30901 L
.11449 .30901 L
.11677 .30901 L
.11922 .30901 L
.12366 .30901 L
.12618 .30901 L
.12894 .30901 L
.13163 .30901 L
.1345 .30901 L
.13719 .30901 L
.13964 .30901 L
.14443 .30901 L
.14927 .30901 L
.15172 .30901 L
.15441 .30901 L
.15916 .30902 L
.16361 .30902 L
.16885 .30902 L
.17364 .30903 L
.17916 .30904 L
Mistroke
.18441 .30905 L
.18945 .30906 L
.19402 .30908 L
.19929 .30911 L
.20425 .30915 L
.20886 .30919 L
.2131 .30925 L
.21775 .30932 L
.22265 .30942 L
.22763 .30956 L
.23216 .30972 L
.23738 .30995 L
.24233 .31024 L
.25112 .31095 L
.25609 .3115 L
.26068 .31213 L
.26532 .31291 L
.27033 .31394 L
.27938 .31644 L
.28462 .31834 L
.29014 .32078 L
.29999 .32652 L
.30957 .33419 L
.31978 .34517 L
.32941 .3587 L
.33816 .37396 L
.35738 .41813 L
.37819 .47997 L
.39771 .54157 L
.40806 .56989 L
.41376 .5827 L
.41916 .5924 L
.42164 .59592 L
.42427 .59897 L
.42564 .60025 L
.42715 .60142 L
.42853 .60225 L
.42983 .60282 L
.43101 .60316 L
.43227 .60332 L
.43334 .6033 L
.43451 .6031 L
.43577 .60269 L
.43648 .60236 L
.43712 .60201 L
.43956 .60014 L
.44074 .59893 L
.442 .59743 L
.44426 .59416 L
.44936 .58404 L
Mistroke
.45376 .57221 L
.45862 .5559 L
.46876 .51114 L
.47804 .45905 L
.49892 .31677 L
.51953 .17408 L
.52956 .11557 L
.53865 .07279 L
.54323 .05551 L
.54818 .04023 L
.55285 .02912 L
.55715 .02173 L
.55952 .01881 L
.56172 .01679 L
.56299 .01595 L
.56415 .01537 L
.56478 .01512 L
.56548 .01492 L
.56674 .01472 L
.56795 .01472 L
.56864 .0148 L
.56929 .01494 L
.5706 .01537 L
.57133 .0157 L
.572 .01606 L
.57437 .01778 L
.57686 .02027 L
.58187 .02726 L
.58642 .03568 L
.59662 .06027 L
.61499 .11617 L
.63581 .18094 L
.655 .22999 L
.67369 .26404 L
.68312 .27629 L
.69348 .28653 L
.70365 .29385 L
.71294 .29866 L
.71808 .30071 L
.72367 .30252 L
.73384 .30495 L
.73918 .30587 L
.7442 .30655 L
.75357 .30749 L
.75883 .30785 L
.76366 .30812 L
.76896 .30834 L
.77456 .30852 L
.77954 .30864 L
.78407 .30873 L
Mistroke
.7893 .30881 L
.79425 .30886 L
.79883 .3089 L
.80305 .30892 L
.80769 .30895 L
.8126 .30896 L
.81751 .30898 L
.82201 .30898 L
.8272 .30899 L
.83213 .309 L
.83666 .309 L
.84087 .309 L
.8455 .30901 L
.85043 .30901 L
.85303 .30901 L
.85578 .30901 L
.8607 .30901 L
.86345 .30901 L
.86601 .30901 L
.86899 .30901 L
.87183 .30901 L
.87426 .30901 L
.87692 .30901 L
.88156 .30901 L
.88419 .30901 L
.88698 .30901 L
.88942 .30901 L
.892 .30901 L
.89433 .30901 L
.89688 .30901 L
.90136 .30901 L
.90389 .30901 L
.90659 .30901 L
.91146 .30901 L
.91382 .30901 L
.91638 .30901 L
.91864 .30901 L
.92103 .30901 L
.92319 .30901 L
.92556 .30901 L
.92973 .30901 L
.93223 .30901 L
.93494 .30901 L
.93734 .30901 L
.93989 .30901 L
.94242 .30901 L
.94471 .30901 L
.94917 .30901 L
.95187 .30901 L
.95434 .30901 L
Mistroke
.95706 .30901 L
.95993 .30901 L
.96243 .30901 L
.96476 .30901 L
.96732 .30901 L
.97006 .30901 L
.97321 .30901 L
.97619 .30901 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[58]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oS`3oool2
000008l0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0
oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo00l0oooo1000
001k0?ooo`00S`3oool2000000l0oooo00@000000?ooo`3oool0oooo0P00001j0?ooo`00S`3oool0
0`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0N03oool0
08l0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
07L0oooo002?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool01P3oool00`000000
oooo0?ooo`1f0?ooo`00S`3oool2000000d0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`1f0?ooo`00S`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0M@3oool008l0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo07D0oooo001e0?ooo`800000103oool2000000D0oooo00<000000?oo
o`3oool00P3oool2000000H0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`030000003oool0oooo07@0oooo001d0?ooo`040000003oool0oooo000000X0oooo00D000000?oo
o`3oool0oooo000000020?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool02P3oool0
0`000000oooo0?ooo`0:0?ooo`030000003oool0oooo07@0oooo001d0?ooo`040000003oool0oooo
000000X0oooo00<000000?ooo`3oool0103oool00`000000oooo0?ooo`030?ooo`<000002@3oool0
0`000000oooo0?ooo`0<0?ooo`030000003oool0oooo07<0oooo001^0?ooo`@000000P3oool01000
0000oooo0?ooo`00000;0?ooo`040000003oool0oooo0?ooo`8000001P3oool00`000000oooo0?oo
o`090?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0L`3oool007@0oooo00@00000
0?ooo`3oool00000203oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00D0oooo
00<000000?ooo`3oool02@3oool00`000000oooo0?ooo`0=0?ooo`030000003oool0oooo0780oooo
001e0?ooo`8000002@3oool4000000<0oooo0`0000050?ooo`030000003oool0oooo00T0oooo00<0
00000?ooo`3oool03@3oool00`000000oooo0?ooo`1b0?ooo`00S`3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo00h0oooo00<000000?ooo`3oool0LP3oool008l0oooo0P0000090?oo
o`030000003oool0oooo00l0oooo00<000000?ooo`3oool0L@3oool008l0oooo00<000000?ooo`3o
ool0203oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo0740oooo002?0?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`1a0?ooo`00S`3o
ool00`000000oooo0?ooo`080?ooo`030000003oool0oooo0100oooo00<000000?ooo`3oool0L03o
ool008l0oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0
oooo0700oooo002?0?ooo`800000203oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo
0700oooo002?0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool04@3oool00`000000
oooo0?ooo`1`0?ooo`00S`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00<00000
3`3oool00`000000oooo0?ooo`1_0?ooo`00S`3oool00`000000oooo0?ooo`060?ooo`030000003o
ool0oooo00D000003P3oool00`000000oooo0?ooo`1_0?ooo`00S`3oool00`000000oooo0?ooo`06
0?ooo`030000003oool0000000H000003@3oool00`000000oooo0?ooo`1_0?ooo`00S`3oool20000
00L0oooo00<000000?ooo`0000001P00000>0?ooo`030000003oool0oooo06h0oooo002?0?ooo`03
0000003oool0oooo00H0oooo00<000000?ooo`0000001P00000>0?ooo`030000003oool0oooo06h0
oooo002?0?ooo`030000003oool0oooo00D0oooo00@000000?ooo`3oool0oooo1@00000?0?ooo`03
0000003oool0oooo06h0oooo002?0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
0P3oool300000140oooo00<000000?ooo`3oool0K@3oool008l0oooo00<000000?ooo`3oool01@3o
ool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo06d0oooo002?0?ooo`8000001P3oool0
0`000000oooo0?ooo`0F0?ooo`030000003oool0oooo06d0oooo002?0?ooo`030000003oool0oooo
00D0oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`1/0?ooo`00S`3oool00`000000
oooo0?ooo`050?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0K03oool007/0oooo
0P0000040?ooo`800000103oool2000000H0oooo00<000000?ooo`3oool0103oool00`000000oooo
0?ooo`0H0?ooo`030000003oool0oooo06`0oooo001j0?ooo`040000003oool0oooo000000P0oooo
00@000000?ooo`3oool000001@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01T0
oooo00<000000?ooo`3oool0J`3oool007X0oooo00@000000?ooo`3oool000002`3oool00`000000
oooo0?ooo`030?ooo`<00000103oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo06/0
oooo001d0?ooo`@000000P3oool010000000oooo0?ooo`0000090?ooo`8000001P3oool00`000000
oooo0?ooo`040?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0J`3oool007X0oooo
00@000000?ooo`3oool000002@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`1Z0?ooo`00N`3oool2000000X0oooo
0`0000050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool06P3oool00`000000oooo
0?ooo`1Z0?ooo`00S`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01X0oooo00<0
00000?ooo`3oool0JP3oool008l0oooo0P0000050?ooo`030000003oool0oooo01/0oooo00<00000
0?ooo`3oool0J@3oool008l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0L0?oo
o`040000003oool0oooo0?ooo`<00000I@3oool008l0oooo00<000000?ooo`3oool00`3oool00`00
0000oooo0?ooo`0L0?ooo`030000003oool0oooo00D00000I03oool008l0oooo00<000000?ooo`3o
ool00`3oool00`000000oooo0?ooo`0M0?ooo`P00000H`3oool008l0oooo00<000000?ooo`3oool0
0`3oool00`000000oooo0?ooo`0M0?ooo`P00000H`3oool008l0oooo0P0000040?ooo`030000003o
ool0oooo01d0oooo2000001S0?ooo`00S`3oool00`000000oooo0?ooo`030?ooo`030000003oool0
oooo01h0oooo1P00001T0?ooo`00S`3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo
01h0oooo00<000000?ooo`0000000P00001U0?ooo`00S`3oool00`000000oooo0?ooo`030?ooo`03
0000003oool0oooo01h0oooo00<000000?ooo`3oool0I`3oool008l0oooo00<000000?ooo`3oool0
0P3oool00`000000oooo0?ooo`0P0?ooo`030000003oool0oooo06H0oooo002?0?ooo`8000000`3o
ool00`000000oooo0?ooo`0P0?ooo`030000003oool0oooo06H0oooo002?0?ooo`030000003oool0
oooo0080oooo00<000000?ooo`3oool0803oool00`000000oooo0?ooo`1V0?ooo`00S`3oool00`00
0000oooo0?ooo`020?ooo`030000003oool0oooo0240oooo00<000000?ooo`3oool0I@3oool008l0
oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0Q0?ooo`030000003oool0oooo06D0
oooo002?0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool08@3oool00`000000oooo
0?ooo`1U0?ooo`00S`3oool2000000<0oooo00<000000?ooo`3oool08P3oool00`000000oooo0?oo
o`1T0?ooo`00S`3oool01@000000oooo0?ooo`3oool0000002D0oooo00<000000?ooo`3oool0I03o
ool007D0oooo0P0000040?ooo`8000000`3oool4000000<0oooo0P0000060?ooo`050000003oool0
oooo0?ooo`0000009P3oool00`000000oooo0?ooo`1S0?ooo`00M03oool010000000oooo0?ooo`00
00080?ooo`030000003oool0oooo00<0oooo00@000000?ooo`3oool000001@3oool01@000000oooo
0?ooo`3oool0000002H0oooo00<000000?ooo`3oool0H`3oool007@0oooo00@000000?ooo`3oool0
00002@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00<0oooo0`0000000`3oool0
00000?ooo`0U0?ooo`030000003oool0oooo06<0oooo001^0?ooo`@000000P3oool010000000oooo
0?ooo`00000:0?ooo`030000003oool0oooo0080oooo0P0000060?ooo`050000003oool0oooo0?oo
o`0000009`3oool00`000000oooo0?ooo`1R0?ooo`00M03oool010000000oooo0?ooo`0000080?oo
o`040000003oool0oooo000000<0oooo00<000000?ooo`3oool01@3oool01@000000oooo0?ooo`3o
ool0000002L0oooo00<000000?ooo`3oool0HP3oool007D0oooo0P00000:0?ooo`800000103oool3
000000D0oooo00D000000?ooo`3oool0oooo0000000X0?ooo`030000003oool0oooo0640oooo002?
0?ooo`040000003oool0oooo000002T0oooo00<000000?ooo`3oool0H@3oool008l0oooo0P000000
0`3oool000000?ooo`0Y0?ooo`030000003oool0oooo0600oooo002?0?ooo`040000003oool0oooo
000002X0oooo00<000000?ooo`3oool0H03oool008l0oooo00@000000?ooo`3oool00000:`3oool0
0`000000oooo0?ooo`1O0?ooo`00S`3oool010000000oooo0?ooo`00000/0?ooo`030000003oool0
oooo05h0oooo002?0?ooo`040000003oool0oooo000002`0oooo00<000000?ooo`3oool0GP3oool0
08l0oooo0`00000^0?ooo`030000003oool0oooo05d0oooo002?0?ooo`030000003oool0000002h0
oooo00<000000?ooo`3oool0G@3oool008l0oooo00<000000?ooo`000000;`3oool00`000000oooo
0?ooo`1L0?ooo`009@3oool300000340oooo1000000b0?ooo`030000003oool0000002l0oooo00<0
00000?ooo`3oool01000000d0?ooo`<000008@3oool002H0oooo00<000000?ooo`3oool0<03oool0
0`000000oooo0?ooo`0c0?ooo`030000003oool000000300oooo00<000000?ooo`000000>03oool0
0`000000oooo0?ooo`0P0?ooo`008`3oool500000380oooo00<000000?ooo`3oool0<P3oool30000
0300oooo00@000000?ooo`3oool00000=03oool500000240oooo000M0?ooo`@000000P3oool01000
0000oooo0?ooo`00000S0?ooo`<000001P3oool4000000@0oooo00<000000?ooo`3oool0<@3oool0
0`000000oooo0000000a0?ooo`040000003oool0oooo000003<0oooo00@000000?ooo`3oool00000
8P3oool002@0oooo00<000000?ooo`0000008P3oool5000000/0oooo00@000000?ooo`3oool00000
<P3oool2000003<0oooo00@000000?ooo`3oool00000<`3oool00`000000oooo0000000R0?ooo`00
9@3oool200000240oooo1`00000;0?ooo`800000<`3oool2000003@0oooo0P00000R0?ooo`<00000
403oool2000001P0oooo0`0000070?ooo`00803oool3000002D0oooo1`0000100?ooo`800000=@3o
ool00`000000oooo0?ooo`0O0?ooo`D00000:03oool5000000H0oooo00090?ooo`<000004`3oool5
000002@0oooo1`0000100?ooo`800000=P3oool2000001h0oooo1`00000L0?ooo`<000001`3oool7
000000D0oooo00080?ooo`D000004@3oool7000002@0oooo1@0000110?ooo`800000>03oool20000
01`0oooo1`00000K0?ooo`D000001P3oool7000000D0oooo00070?ooo`L00000403oool7000002D0
oooo0`0000120?ooo`800000>P3oool2000001X0oooo1`00000J0?ooo`L000001@3oool7000000D0
oooo00070?ooo`L00000403oool7000006X0oooo0P00000l0?ooo`@000005`3oool5000001/0oooo
1`0000060?ooo`D000001P3oool000L0oooo1`0000030?ooo`<000002`3oool5000006/0oooo00<0
00000?ooo`3oool0?`3oool600000180oooo0`00000L0?ooo`L00000103oool6000000L0oooo003o
0000020000000@3oool000H0oooo1P0000030?ooo`L00000303oool00`000000oooo0?ooo`0;0?oo
o`030000003oool0oooo00<000001`3oool00`000000oooo0?ooo`070?ooo`P000002@3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool02P3oool00`00
0000oooo0?ooo`0;0?ooo`030000003oool0oooo00X0oooo00<000000?ooo`3oool02P3oool50000
00X0oooo0`0000040?ooo`L00000203oool000D0oooo1@0000050?ooo`L00000303oool00`000000
oooo0?ooo`0=0?ooo`D00000603oool2000000L0oooo00<000000?ooo`3oool0=03oool00`000000
oooo0?ooo`0d0?ooo`030000003oool0oooo0340oooo1`00000@0?ooo`L00000203oool000@0oooo
1`0000040?ooo`L000006`3oool7000001T0oooo0`00000k0?ooo`030000003oool0oooo0400oooo
0`00000U0?ooo`L00000403oool7000000P0oooo00040?ooo`L000001@3oool5000001`0oooo1`00
000L0?ooo`800000>@3oool00`000000oooo0?ooo`0o0?ooo`D00000903oool700000140oooo1@00
00090?ooo`00103oool7000000H0oooo0`00000M0?ooo`L000007P3oool00`000000oooo0?ooo`0f
0?ooo`800000?`3oool7000002@0oooo1@00000C0?ooo`<000002P3oool000D0oooo1@00000X0?oo
o`D00000803oool00`000000oooo0?ooo`0d0?ooo`800000@03oool7000002D0oooo0`00000Q0?oo
o`001P3oool3000002X0oooo0`00000R0?ooo`030000003oool0oooo03<0oooo0P0000100?ooo`L0
0000B@3oool005T0oooo00<000000?ooo`3oool0<P3oool200000440oooo1@00001:0?ooo`00FP3o
ool00`000000oooo0?ooo`0a0?ooo`800000@P3oool3000004/0oooo001K0?ooo`030000003oool0
oooo0300oooo0`00002?0?ooo`00G03oool00`000000oooo0?ooo`0_0?ooo`800000T03oool005`0
oooo00<000000?ooo`3oool0;`3oool200000900oooo001M0?ooo`030000003oool0oooo02d0oooo
00<000000?ooo`000000T03oool005h0oooo00<000000?ooo`3oool0;03oool00`000000oooo0000
002@0?ooo`00G`3oool00`000000oooo0?ooo`0[0?ooo`040000003oool00000000008l0oooo001O
0?ooo`030000003oool0oooo02/0oooo00<000000?ooo`000000T03oool00600oooo00<000000?oo
o`3oool0:P3oool00`000000oooo0000002@0?ooo`00H03oool00`000000oooo0?ooo`0Z0?ooo`03
0000003oool000000900oooo001Q0?ooo`030000003oool0oooo02T0oooo00<000000?ooo`000000
T03oool00640oooo00<000000?ooo`3oool0:03oool00`000000oooo0?ooo`02000008l0oooo001R
0?ooo`030000003oool0oooo02L0oooo00@000000?ooo`3oool00000T03oool00680oooo00<00000
0?ooo`3oool09`3oool010000000oooo0?ooo`00002@0?ooo`00H`3oool00`000000oooo0?ooo`0?
0?ooo`800000103oool2000000<0oooo100000030?ooo`8000000`3oool010000000oooo0?ooo`00
002@0?ooo`00H`3oool00`000000oooo0?ooo`0>0?ooo`040000003oool0oooo000000P0oooo00<0
00000?ooo`3oool00`3oool010000000oooo0?ooo`0000020?ooo`040000003oool0oooo00000900
oooo001S0?ooo`030000003oool0oooo00h0oooo00@000000?ooo`3oool000002@3oool00`000000
oooo0?ooo`050?ooo`040000003oool0oooo00000080oooo0`00002>0?ooo`00I03oool00`000000
oooo0?ooo`0=0?ooo`040000003oool0oooo000000X0oooo00<000000?ooo`3oool00P3oool20000
0080oooo00D000000?ooo`3oool0oooo0000002@0?ooo`00I03oool00`000000oooo0?ooo`0=0?oo
o`040000003oool0oooo000000P0oooo00@000000?ooo`3oool000000`3oool01@000000oooo0?oo
o`3oool0000000<0oooo00<000000?ooo`3oool0SP3oool006D0oooo00<000000?ooo`3oool03@3o
ool2000000X0oooo0P0000040?ooo`<0000000<0oooo0000003oool00P3oool00`000000oooo0?oo
o`2>0?ooo`00I@3oool00`000000oooo0?ooo`0S0?ooo`040000003oool0oooo0?ooo`800000S`3o
ool006H0oooo00<000000?ooo`3oool08P3oool01@000000oooo0?ooo`3oool000000900oooo001V
0?ooo`030000003oool0oooo0280oooo00D000000?ooo`3oool0oooo0000002@0?ooo`00IP3oool0
0`000000oooo0?ooo`0R0?ooo`050000003oool0oooo0?ooo`000000T03oool006L0oooo00<00000
0?ooo`3oool0803oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo08h0oooo001W0?oo
o`030000003oool0oooo0200oooo00<000000?ooo`3oool00P3oool2000008l0oooo001W0?ooo`03
0000003oool0oooo0200oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`2>0?ooo`00
J03oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0
SP3oool006@0oooo0`0000000`3oool000000?ooo`0P0?ooo`030000003oool0oooo0080oooo00<0
00000?ooo`3oool0SP3oool006<0oooo1P00000Q0?ooo`030000003oool0oooo0080oooo00<00000
0?ooo`3oool0SP3oool00680oooo2000000P0?ooo`030000003oool0oooo0080oooo0P00002?0?oo
o`00HP3oool8000001l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`2>0?ooo`00
HP3oool8000001l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`2>0?ooo`00H`3o
ool500000080oooo00<000000?ooo`3oool0703oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo08h0oooo001T0?ooo`<000000`3oool00`000000oooo0?ooo`0L0?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool0SP3oool006X0oooo00<000000?ooo`3oool0703oool00`00
0000oooo0?ooo`030?ooo`800000S`3oool006/0oooo00<000000?ooo`3oool06`3oool00`000000
oooo0?ooo`030?ooo`030000003oool0oooo08h0oooo001[0?ooo`030000003oool0oooo01X0oooo
00<000000?ooo`3oool0103oool00`000000oooo0?ooo`2>0?ooo`00J`3oool00`000000oooo0?oo
o`0=0?ooo`800000103oool2000000@0oooo0P0000060?ooo`030000003oool0oooo08h0oooo001/
0?ooo`030000003oool0oooo00/0oooo00@000000?ooo`3oool00000203oool010000000oooo0000
000000050?ooo`030000003oool0oooo08h0oooo001/0?ooo`030000003oool0oooo00/0oooo00@0
00000?ooo`3oool000002P3oool2000000D0oooo0`00002>0?ooo`00K03oool00`000000oooo0?oo
o`0;0?ooo`040000003oool0oooo000000T0oooo0P0000060?ooo`030000003oool0oooo08h0oooo
001]0?ooo`030000003oool0oooo00X0oooo00@000000?ooo`3oool000002@3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo08h0oooo001]0?ooo`030000003oool0oooo00/0oooo0P00
000:0?ooo`<000001@3oool00`000000oooo0?ooo`2>0?ooo`00K@3oool00`000000oooo0?ooo`0G
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0SP3oool006h0oooo00<000000?oo
o`3oool05P3oool00`000000oooo0?ooo`050?ooo`800000S`3oool006h0oooo00<000000?ooo`3o
ool05P3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo08h0oooo001^0?ooo`030000
003oool0oooo00l0oooo0`0000030?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0
SP3oool006l0oooo00<000000?ooo`3oool03@3oool500000080oooo00<000000?ooo`3oool01P3o
ool00`000000oooo0?ooo`2>0?ooo`00K`3oool00`000000oooo0?ooo`0<0?ooo`L0000000<0oooo
0000003oool01`3oool00`000000oooo0?ooo`2>0?ooo`00K`3oool00`000000oooo0?ooo`0<0?oo
o`L0000000<0oooo0000003oool01`3oool2000008l0oooo001`0?ooo`030000003oool0oooo00/0
oooo1`0000000`3oool000000?ooo`070?ooo`030000003oool0oooo08h0oooo001`0?ooo`030000
003oool0oooo00`0oooo1@0000000`3oool000000?ooo`080?ooo`030000003oool0oooo08h0oooo
001`0?ooo`030000003oool0oooo00d0oooo0`0000020?ooo`030000003oool0oooo00L0oooo00<0
00000?ooo`3oool0SP3oool00740oooo00<000000?ooo`3oool04@3oool00`000000oooo0?ooo`07
0?ooo`030000003oool0oooo08h0oooo001a0?ooo`030000003oool0oooo0140oooo00<000000?oo
o`3oool01`3oool2000008l0oooo001a0?ooo`030000003oool0oooo0100oooo00<000000?ooo`3o
ool0203oool00`000000oooo0?ooo`2>0?ooo`00LP3oool00`000000oooo0?ooo`0?0?ooo`030000
003oool0oooo00P0oooo00<000000?ooo`3oool0SP3oool00780oooo00<000000?ooo`3oool03`3o
ool00`000000oooo0?ooo`080?ooo`030000003oool0oooo08h0oooo001b0?ooo`030000003oool0
oooo00l0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`2>0?ooo`00L`3oool00`00
0000oooo0?ooo`0=0?ooo`030000003oool0oooo00T0oooo0P00002?0?ooo`00L`3oool00`000000
oooo0?ooo`0=0?ooo`030000003oool0oooo00T0oooo00<000000?ooo`3oool0SP3oool007<0oooo
00<000000?ooo`3oool03@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo08h0oooo
001d0?ooo`<00000103oool2000000D0oooo0P0000030?ooo`8000001P3oool00`000000oooo0?oo
o`2>0?ooo`00M03oool010000000oooo0?ooo`00000:0?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo08h0oooo001d0?ooo`80000000<0
oooo0000003oool02@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo00<0oooo0`00
002>0?ooo`00M03oool2000000030?ooo`000000oooo00T0oooo0P0000030?ooo`8000001P3oool0
0`000000oooo0?ooo`2>0?ooo`00M03oool2000000030?ooo`000000oooo00L0oooo00@000000?oo
o`00000000000`3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo08h0oooo001e0?oo
o`8000002@3oool4000000<0oooo0`0000050?ooo`030000003oool0oooo08h0oooo001f0?ooo`03
0000003oool0oooo00P0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`2>0?ooo`00
M`3oool00`000000oooo0?ooo`070?ooo`030000003oool0oooo00/0oooo0P00002?0?ooo`00M`3o
ool00`000000oooo0?ooo`060?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool0SP3o
ool007P0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0<0?ooo`030000003oool0
oooo08h0oooo001i0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool03@3oool00`00
0000oooo0?ooo`2>0?ooo`00N@3oool2000000<0oooo00<000000?ooo`3oool03P3oool00`000000
oooo0?ooo`2>0?ooo`00NP3oool400000140oooo0P00002?0?ooo`00S`3oool00`000000oooo0?oo
o`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?oo
o`00S`3oool2000008l0oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-5.26302, -0.90059, \
0.0366761, 0.0101799}}]
}, Open  ]],

Cell["\<\
This shows a plot of the derivative computed using a uniform grid \
with the same number of points with maximal difference order.\
\>", \
"MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(ugrid\  = \ \(-5\)\  + \ 
          10. \ Range[0, 15]/15;\)\), "\[IndentingNewLine]", 
    \(\(Show[
        Block[{$DisplayFunction\  = \ 
              Identity}, \ {\[IndentingNewLine]ListPlot[
              Transpose[{ugrid, \ 
                  NDSolve`FiniteDifferenceDerivative[1, ugrid, \ 
                    Exp[\(-ugrid\^2\)], 
                    DifferenceOrder \[Rule] Length[ugrid]]}], \ 
              PlotStyle \[Rule] PointSize[0.025]], \[IndentingNewLine]Plot[
              Evaluate[D[Exp[\(-x\^2\)], x]], {x, \(-5\), 5}]}], \ 
        PlotRange \[Rule] All];\)\)}], "Input",
  CellLabel->"In[59]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[BoxData[
    \(NDSolve`FiniteDifferenceDerivative::"ordred" \(\(:\)\(\ \)\) 
      "There are insufficient points in dimension \!\(1\) to achieve the \
requested approximation order. Order will be reduced to \!\(15\)."\)], \
"Message",
  CellLabel->"From In[59]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.0952381 0.309017 0.0107313 [
[.11905 .29652 -6 -9 ]
[.11905 .29652 6 0 ]
[.30952 .29652 -6 -9 ]
[.30952 .29652 6 0 ]
[.69048 .29652 -3 -9 ]
[.69048 .29652 3 0 ]
[.88095 .29652 -3 -9 ]
[.88095 .29652 3 0 ]
[.4875 .09439 -18 -4.5 ]
[.4875 .09439 0 4.5 ]
[.4875 .2017 -18 -4.5 ]
[.4875 .2017 0 4.5 ]
[.4875 .41633 -12 -4.5 ]
[.4875 .41633 0 4.5 ]
[.4875 .52364 -12 -4.5 ]
[.4875 .52364 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.11905 .30902 m
.11905 .31527 L
s
[(-4)] .11905 .29652 0 1 Mshowa
.30952 .30902 m
.30952 .31527 L
s
[(-2)] .30952 .29652 0 1 Mshowa
.69048 .30902 m
.69048 .31527 L
s
[(2)] .69048 .29652 0 1 Mshowa
.88095 .30902 m
.88095 .31527 L
s
[(4)] .88095 .29652 0 1 Mshowa
.125 Mabswid
.16667 .30902 m
.16667 .31277 L
s
.21429 .30902 m
.21429 .31277 L
s
.2619 .30902 m
.2619 .31277 L
s
.35714 .30902 m
.35714 .31277 L
s
.40476 .30902 m
.40476 .31277 L
s
.45238 .30902 m
.45238 .31277 L
s
.54762 .30902 m
.54762 .31277 L
s
.59524 .30902 m
.59524 .31277 L
s
.64286 .30902 m
.64286 .31277 L
s
.7381 .30902 m
.7381 .31277 L
s
.78571 .30902 m
.78571 .31277 L
s
.83333 .30902 m
.83333 .31277 L
s
.07143 .30902 m
.07143 .31277 L
s
.02381 .30902 m
.02381 .31277 L
s
.92857 .30902 m
.92857 .31277 L
s
.97619 .30902 m
.97619 .31277 L
s
.25 Mabswid
0 .30902 m
1 .30902 L
s
.5 .09439 m
.50625 .09439 L
s
[(-20)] .4875 .09439 1 0 Mshowa
.5 .2017 m
.50625 .2017 L
s
[(-10)] .4875 .2017 1 0 Mshowa
.5 .41633 m
.50625 .41633 L
s
[(10)] .4875 .41633 1 0 Mshowa
.5 .52364 m
.50625 .52364 L
s
[(20)] .4875 .52364 1 0 Mshowa
.125 Mabswid
.5 .11585 m
.50375 .11585 L
s
.5 .13732 m
.50375 .13732 L
s
.5 .15878 m
.50375 .15878 L
s
.5 .18024 m
.50375 .18024 L
s
.5 .22317 m
.50375 .22317 L
s
.5 .24463 m
.50375 .24463 L
s
.5 .26609 m
.50375 .26609 L
s
.5 .28755 m
.50375 .28755 L
s
.5 .33048 m
.50375 .33048 L
s
.5 .35194 m
.50375 .35194 L
s
.5 .3734 m
.50375 .3734 L
s
.5 .39487 m
.50375 .39487 L
s
.5 .43779 m
.50375 .43779 L
s
.5 .45926 m
.50375 .45926 L
s
.5 .48072 m
.50375 .48072 L
s
.5 .50218 m
.50375 .50218 L
s
.5 .07293 m
.50375 .07293 L
s
.5 .05147 m
.50375 .05147 L
s
.5 .03 m
.50375 .03 L
s
.5 .00854 m
.50375 .00854 L
s
.5 .54511 m
.50375 .54511 L
s
.5 .56657 m
.50375 .56657 L
s
.5 .58803 m
.50375 .58803 L
s
.5 .60949 m
.50375 .60949 L
s
.25 Mabswid
.5 0 m
.5 .61803 L
s
.025 w
.02381 .60332 Mdot
.0873 .2837 Mdot
.15079 .31382 Mdot
.21429 .30751 Mdot
.27778 .30994 Mdot
.34127 .31079 Mdot
.40476 .31727 Mdot
.46825 .31509 Mdot
.53175 .30294 Mdot
.59524 .30076 Mdot
.65873 .30724 Mdot
.72222 .30809 Mdot
.78571 .31053 Mdot
.84921 .30421 Mdot
.9127 .33433 Mdot
.97619 .01472 Mdot
.5 Mabswid
.02381 .30902 m
.02605 .30902 L
.02846 .30902 L
.03279 .30902 L
.03543 .30902 L
.03793 .30902 L
.04018 .30902 L
.04262 .30902 L
.045 .30902 L
.04762 .30902 L
.05036 .30902 L
.05293 .30902 L
.05751 .30902 L
.06009 .30902 L
.06244 .30902 L
.06504 .30902 L
.0678 .30902 L
.07041 .30902 L
.07277 .30902 L
.07556 .30902 L
.07858 .30902 L
.08143 .30902 L
.08405 .30902 L
.08881 .30902 L
.09151 .30902 L
.09403 .30902 L
.09912 .30902 L
.10198 .30902 L
.10458 .30902 L
.10939 .30902 L
.11181 .30902 L
.11449 .30902 L
.11677 .30902 L
.11922 .30902 L
.12366 .30902 L
.12618 .30902 L
.12894 .30902 L
.13163 .30902 L
.1345 .30902 L
.13719 .30902 L
.13964 .30902 L
.14443 .30902 L
.14927 .30902 L
.15172 .30902 L
.15441 .30902 L
.15916 .30902 L
.16361 .30902 L
.16885 .30902 L
.17364 .30902 L
.17916 .30902 L
Mistroke
.18441 .30902 L
.18945 .30902 L
.19402 .30902 L
.19929 .30902 L
.20425 .30902 L
.20886 .30902 L
.2131 .30902 L
.21775 .30903 L
.22265 .30903 L
.22763 .30903 L
.23216 .30904 L
.23738 .30905 L
.24233 .30906 L
.25112 .30908 L
.25609 .30909 L
.26068 .30911 L
.26532 .30914 L
.27033 .30917 L
.27938 .30925 L
.28462 .30931 L
.29014 .30939 L
.29999 .30956 L
.30957 .3098 L
.31978 .31015 L
.32941 .31057 L
.33816 .31105 L
.35738 .31243 L
.37819 .31436 L
.39771 .31629 L
.40806 .31718 L
.41376 .31758 L
.41916 .31788 L
.42164 .31799 L
.42427 .31809 L
.42564 .31813 L
.42715 .31816 L
.42853 .31819 L
.42983 .31821 L
.43101 .31822 L
.43227 .31822 L
.43334 .31822 L
.43451 .31821 L
.43577 .3182 L
.43648 .31819 L
.43712 .31818 L
.43956 .31812 L
.44074 .31808 L
.442 .31804 L
.44426 .31794 L
.44936 .31762 L
Mistroke
.45376 .31725 L
.45862 .31674 L
.46876 .31534 L
.47804 .31371 L
.49892 .30926 L
.51953 .3048 L
.52956 .30297 L
.53865 .30163 L
.54323 .30109 L
.54818 .30061 L
.55285 .30026 L
.55715 .30003 L
.55952 .29994 L
.56172 .29988 L
.56299 .29985 L
.56415 .29983 L
.56478 .29983 L
.56548 .29982 L
.56674 .29981 L
.56795 .29981 L
.56864 .29982 L
.56929 .29982 L
.5706 .29983 L
.57133 .29984 L
.572 .29986 L
.57437 .29991 L
.57686 .29999 L
.58187 .30021 L
.58642 .30047 L
.59662 .30124 L
.61499 .30299 L
.63581 .30501 L
.655 .30655 L
.67369 .30761 L
.68312 .30799 L
.69348 .30831 L
.70365 .30854 L
.71294 .30869 L
.71808 .30876 L
.72367 .30881 L
.73384 .30889 L
.73918 .30892 L
.7442 .30894 L
.75357 .30897 L
.75883 .30898 L
.76366 .30899 L
.76896 .309 L
.77456 .309 L
.77954 .30901 L
.78407 .30901 L
Mistroke
.7893 .30901 L
.79425 .30901 L
.79883 .30901 L
.80305 .30901 L
.80769 .30901 L
.8126 .30902 L
.81751 .30902 L
.82201 .30902 L
.8272 .30902 L
.83213 .30902 L
.83666 .30902 L
.84087 .30902 L
.8455 .30902 L
.85043 .30902 L
.85303 .30902 L
.85578 .30902 L
.8607 .30902 L
.86345 .30902 L
.86601 .30902 L
.86899 .30902 L
.87183 .30902 L
.87426 .30902 L
.87692 .30902 L
.88156 .30902 L
.88419 .30902 L
.88698 .30902 L
.88942 .30902 L
.892 .30902 L
.89433 .30902 L
.89688 .30902 L
.90136 .30902 L
.90389 .30902 L
.90659 .30902 L
.91146 .30902 L
.91382 .30902 L
.91638 .30902 L
.91864 .30902 L
.92103 .30902 L
.92319 .30902 L
.92556 .30902 L
.92973 .30902 L
.93223 .30902 L
.93494 .30902 L
.93734 .30902 L
.93989 .30902 L
.94242 .30902 L
.94471 .30902 L
.94917 .30902 L
.95187 .30902 L
.95434 .30902 L
Mistroke
.95706 .30902 L
.95993 .30902 L
.96243 .30902 L
.96476 .30902 L
.96732 .30902 L
.97006 .30902 L
.97321 .30902 L
.97619 .30902 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[59]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oS`3oool0
0`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`240?ooo`<000001`3oool008l0
oooo0P0000240?ooo`D000001P3oool008l0oooo00<000000?ooo`3oool0PP3oool7000000D0oooo
002?0?ooo`030000003oool0oooo0880oooo1`0000050?ooo`00S`3oool00`000000oooo0?ooo`22
0?ooo`L000001@3oool008l0oooo00<000000?ooo`3oool0P`3oool5000000H0oooo002?0?ooo`03
0000003oool0oooo08@0oooo0`0000070?ooo`00S`3oool2000008l0oooo002?0?ooo`030000003o
ool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo
08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo
002?0?ooo`800000S`3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?oo
o`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0
SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo0P00002?0?ooo`00S`3oool00`00
0000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo
0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00P03oool4000000<0oooo0P000006
0?ooo`030000003oool0oooo08h0oooo00200?ooo`030000003oool0oooo00<0oooo00@000000?oo
o`3oool000001@3oool00`000000oooo0?ooo`2>0?ooo`00P@3oool00`000000oooo0?ooo`020?oo
o`040000003oool0oooo000000D0oooo0`00002>0?ooo`00NP3oool4000000@0oooo00D000000?oo
o`3oool0oooo000000020?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0SP3oool0
0800oooo00@000000?ooo`3oool000000P3oool010000000oooo0?ooo`0000050?ooo`030000003o
ool0oooo08h0oooo00210?ooo`800000103oool2000000H0oooo00<000000?ooo`3oool0SP3oool0
08l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo
0P00002?0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>
0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00
S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool2000008l0oooo002?0?ooo`030000003oool0
oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0
oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?
0?ooo`800000S`3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3o
ool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3o
ool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo0P00002?0?ooo`00S`3oool00`000000
oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?oo
o`2>0?ooo`00P03oool500000080oooo0P0000060?ooo`030000003oool0oooo08h0oooo00220?oo
o`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`030?ooo`030000003oool0
oooo08h0oooo00220?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo0?ooo`03
0?ooo`<00000SP3oool007X0oooo100000040?ooo`050000003oool0oooo0?ooo`0000000P3oool0
0`000000oooo0?ooo`030?ooo`030000003oool0oooo08h0oooo00220?ooo`050000003oool0oooo
0?ooo`0000000P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo08h0oooo00210?oo
o`800000103oool2000000H0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3o
ool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo0P00002?0?ooo`00S`3oool0
0`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000
oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?oo
o`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool2000008l0oooo002?0?ooo`03
0000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003o
ool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo
08h0oooo000U0?ooo`<00000<@3oool400000380oooo0P00000c0?ooo`@00000=03oool300000240
oooo000V0?ooo`030000003oool0oooo0300oooo00<000000?ooo`3oool0<`3oool00`000000oooo
0?ooo`0b0?ooo`030000003oool0oooo03H0oooo00<000000?ooo`3oool0803oool001P0oooo0`00
00080?ooo`D00000<P3oool00`000000oooo0?ooo`0b0?ooo`030000003oool0oooo03<0oooo00<0
00000?ooo`3oool0<P3oool500000240oooo000G0?ooo`D0000000<0oooo0000000000000P000002
0?ooo`040000003oool0oooo000002`0oooo100000040?ooo`030000003oool0oooo0340oooo00<0
00000?ooo`3oool0=03oool00`000000oooo0?ooo`0a0?ooo`040000003oool0oooo00000280oooo
000F0?ooo`L000001`3oool00`000000oooo0000000b0?ooo`040000003oool0oooo00000380oooo
00<000000?ooo`3oool0<P3oool010000000oooo0?ooo`00000c0?ooo`030000003oool000000280
oooo000F0?ooo`L00000203oool2000003<0oooo0P00000c0?ooo`030000003oool0oooo03<0oooo
0P00000e0?ooo`8000008P3oool001H0oooo1`00001b0?ooo`800000S`3oool001L0oooo1@00001c
0?ooo`030000003oool0oooo00D0oooo0`00000?0?ooo`<00000M03oool001P0oooo0`00001d0?oo
o`030000003oool0oooo00@0oooo1@00000=0?ooo`D00000A@3oool3000002/0oooo000l0?ooo`<0
00003`3oool3000003h0oooo00<000000?ooo`3oool00`3oool7000000<0oooo1`0000000`3oool0
000000000005000000d0oooo0`00000@0?ooo`<00000803oool5000002X0oooo000Z0?ooo`<00000
3P3oool5000000d0oooo1@00000>0?ooo`<00000;03oool00`000000oooo0?ooo`030?ooo`X00000
1`3oool=000000L0oooo1@00000>0?ooo`D000003P3oool3000000d0oooo1`00000Y0?ooo`00:@3o
ool5000000`0oooo1`00000;0?ooo`L00000303oool5000000l0oooo0`00000?0?ooo`<000001`3o
ool010000000oooo0?ooo`3oool9000000/0oooo1`0000050?ooo`d00000303oool7000000`0oooo
1@00000<0?ooo`L00000:@3oool00?l00000800000010?ooo`001`3oool00`000000oooo0?ooo`0:
0?ooo`030000003oool0oooo00/0oooo00<000000?ooo`3oool00`3oool7000000030?ooo`000000
oooo00P0oooo1`00000:0?ooo`P000001@3oool00`000000oooo0?ooo`030?ooo`/00000203oool7
000000T0oooo00<000000?ooo`0000002@0000020?ooo`030000003oool0oooo00D0oooo0`000003
0?ooo`030000003oool0oooo00T0oooo0`00000<0?ooo`P000001P3oool00`000000oooo0?ooo`03
0?ooo`P000002P3oool7000000T0oooo00<000000?ooo`3oool01@0000060?ooo`030000003oool0
oooo00/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`060?ooo`008P3oool00`00
0000oooo0?ooo`030?ooo`L00000303oool5000000d0oooo1@0000060?ooo`030000003oool0oooo
00<0oooo1`0000040?ooo`l000002@3oool9000000D0oooo00<000000?ooo`3oool0:03oool50000
00L0oooo00<000000?ooo`3oool0103oool5000000`0oooo1`00000=0?ooo`<000001`3oool00`00
0000oooo0?ooo`0Q0?ooo`00:@3oool5000000h0oooo0`00000?0?ooo`<000003P3oool5000000d0
oooo400000020?ooo`L000001@3oool00`000000oooo0?ooo`0Y0?ooo`<00000403oool3000000h0
oooo1@00000l0?ooo`00:P3oool3000003<0oooo0`00000?0?ooo`D000003@3oool5000000H0oooo
00<000000?ooo`3oool0CP3oool300000240oooo0`00000I0?ooo`00L`3oool3000000l0oooo0`00
00070?ooo`030000003oool0oooo0740oooo1@00000H0?ooo`00S`3oool200000740oooo1`00000G
0?ooo`00S`3oool00`000000oooo0?ooo`1`0?ooo`L000005`3oool008l0oooo00<000000?ooo`3o
ool0L03oool7000001L0oooo002?0?ooo`030000003oool0oooo0740oooo1@00000H0?ooo`00S`3o
ool00`000000oooo0?ooo`1b0?ooo`<000006@3oool008l0oooo00<000000?ooo`3oool0SP3oool0
08l0oooo0P00002?0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo
0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>
0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool2000008l0oooo002?0?ooo`030000
003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0
oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0
oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`800000S`3oool008l0oooo00<00000
0?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3o
ool0SP3oool00800oooo1@0000020?ooo`8000001P3oool00`000000oooo0?ooo`2>0?ooo`00PP3o
ool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool00`3oool00`000000oooo
0?ooo`2>0?ooo`00PP3oool01@000000oooo0?ooo`3oool000000080oooo00<000000?ooo`3oool0
0`3oool3000008h0oooo00220?ooo`050000003oool0oooo0?ooo`0000000P3oool00`000000oooo
0?ooo`030?ooo`030000003oool0oooo08h0oooo00220?ooo`050000003oool0oooo0?ooo`000000
0P3oool00`000000oooo0?ooo`030?ooo`030000003oool0oooo08h0oooo00210?ooo`800000103o
ool2000000H0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool0
08l0oooo00<000000?ooo`3oool0SP3oool008l0oooo0P00002?0?ooo`00S`3oool00`000000oooo
0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>
0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00
S`3oool2000008l0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0
oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0
oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`800000S`3oool008l0oooo00<00000
0?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3o
ool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3o
ool008l0oooo0P00002?0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000
oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00P03oool4000000<0oooo0P00
00060?ooo`030000003oool0oooo08h0oooo00200?ooo`030000003oool0oooo00<0oooo00@00000
0?ooo`3oool000001@3oool00`000000oooo0?ooo`2>0?ooo`00P@3oool00`000000oooo0?ooo`02
0?ooo`040000003oool0oooo000000D0oooo0`00002>0?ooo`00PP3oool01@000000oooo0?ooo`3o
ool000000080oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`2>0?ooo`00P03oool0
10000000oooo0?ooo`0000020?ooo`040000003oool0oooo000000D0oooo00<000000?ooo`3oool0
SP3oool00840oooo0P0000040?ooo`8000001P3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool0
0`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000
oooo0?ooo`2>0?ooo`00S`3oool2000008l0oooo002?0?ooo`030000003oool0oooo08h0oooo002?
0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`03
0000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`800000S`3o
ool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0
oooo00<000000?ooo`3oool0SP3oool008l0oooo00<000000?ooo`3oool0SP3oool008l0oooo00<0
00000?ooo`3oool0SP3oool008l0oooo0P00002?0?ooo`001P3oool3000008H0oooo00<000000?oo
o`3oool0SP3oool000D0oooo1@0000250?ooo`030000003oool0oooo08h0oooo00040?ooo`L00000
Q03oool00`000000oooo0?ooo`2>0?ooo`00103oool7000008@0oooo00<000000?ooo`3oool0SP3o
ool000@0oooo1`0000240?ooo`030000003oool0oooo08h0oooo00050?ooo`D00000Q@3oool20000
08l0oooo00060?ooo`<00000QP3oool00`000000oooo0?ooo`2>0?ooo`00S`3oool00`000000oooo
0?ooo`2>0?ooo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-5.26302, -28.7961, \
0.0366761, 0.325493}}]
}, Open  ]],

Cell["\<\
Even though the approximation is somewhat better in the center \
(because the points are more closely spaced there in the uniform grid), the \
plot clearly shows the disastrous oscillation typical of overly high order \
finite difference approximations. Using the Chebyshev-Gauss-Lobatto spacing \
has minimized this.\
\>", "Text",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell["\<\
This shows a plot of the pseudospectral derivative approximation \
computed using a uniform grid with periodic repetition.\
\>", "MathCaption",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(ugrid\  = \ \(-5\)\  + \ 
          10. \ Range[0, 15]/15;\)\), "\[IndentingNewLine]", 
    \(\(Show[
        Block[{$DisplayFunction\  = \ 
              Identity}, \ {\[IndentingNewLine]ListPlot[
              Transpose[{ugrid, \ 
                  NDSolve`FiniteDifferenceDerivative[1, ugrid, \ 
                    Exp[\(-ugrid\^2\)], 
                    DifferenceOrder \[Rule] "\<Pseudospectral\>", \ 
                    PeriodicInterpolation \[Rule] True]}], \ 
              PlotStyle \[Rule] PointSize[0.025]], \[IndentingNewLine]Plot[
              Evaluate[D[Exp[\(-x\^2\)], x]], {x, \(-5\), 5}]}], \ 
        PlotRange \[Rule] All];\)\)}], "Input",
  CellLabel->"In[61]:=",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.5 0.0952381 0.30901 0.343123 [
[.11905 .29651 -6 -9 ]
[.11905 .29651 6 0 ]
[.30952 .29651 -6 -9 ]
[.30952 .29651 6 0 ]
[.69048 .29651 -3 -9 ]
[.69048 .29651 3 0 ]
[.88095 .29651 -3 -9 ]
[.88095 .29651 3 0 ]
[.4875 .05167 -30 -4.5 ]
[.4875 .05167 0 4.5 ]
[.4875 .13745 -24 -4.5 ]
[.4875 .13745 0 4.5 ]
[.4875 .22323 -30 -4.5 ]
[.4875 .22323 0 4.5 ]
[.4875 .39479 -24 -4.5 ]
[.4875 .39479 0 4.5 ]
[.4875 .48057 -18 -4.5 ]
[.4875 .48057 0 4.5 ]
[.4875 .56635 -24 -4.5 ]
[.4875 .56635 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.11905 .30901 m
.11905 .31526 L
s
[(-4)] .11905 .29651 0 1 Mshowa
.30952 .30901 m
.30952 .31526 L
s
[(-2)] .30952 .29651 0 1 Mshowa
.69048 .30901 m
.69048 .31526 L
s
[(2)] .69048 .29651 0 1 Mshowa
.88095 .30901 m
.88095 .31526 L
s
[(4)] .88095 .29651 0 1 Mshowa
.125 Mabswid
.16667 .30901 m
.16667 .31276 L
s
.21429 .30901 m
.21429 .31276 L
s
.2619 .30901 m
.2619 .31276 L
s
.35714 .30901 m
.35714 .31276 L
s
.40476 .30901 m
.40476 .31276 L
s
.45238 .30901 m
.45238 .31276 L
s
.54762 .30901 m
.54762 .31276 L
s
.59524 .30901 m
.59524 .31276 L
s
.64286 .30901 m
.64286 .31276 L
s
.7381 .30901 m
.7381 .31276 L
s
.78571 .30901 m
.78571 .31276 L
s
.83333 .30901 m
.83333 .31276 L
s
.07143 .30901 m
.07143 .31276 L
s
.02381 .30901 m
.02381 .31276 L
s
.92857 .30901 m
.92857 .31276 L
s
.97619 .30901 m
.97619 .31276 L
s
.25 Mabswid
0 .30901 m
1 .30901 L
s
.5 .05167 m
.50625 .05167 L
s
[(-0.75)] .4875 .05167 1 0 Mshowa
.5 .13745 m
.50625 .13745 L
s
[(-0.5)] .4875 .13745 1 0 Mshowa
.5 .22323 m
.50625 .22323 L
s
[(-0.25)] .4875 .22323 1 0 Mshowa
.5 .39479 m
.50625 .39479 L
s
[(0.25)] .4875 .39479 1 0 Mshowa
.5 .48057 m
.50625 .48057 L
s
[(0.5)] .4875 .48057 1 0 Mshowa
.5 .56635 m
.50625 .56635 L
s
[(0.75)] .4875 .56635 1 0 Mshowa
.125 Mabswid
.5 .06882 m
.50375 .06882 L
s
.5 .08598 m
.50375 .08598 L
s
.5 .10314 m
.50375 .10314 L
s
.5 .12029 m
.50375 .12029 L
s
.5 .1546 m
.50375 .1546 L
s
.5 .17176 m
.50375 .17176 L
s
.5 .18892 m
.50375 .18892 L
s
.5 .20607 m
.50375 .20607 L
s
.5 .24039 m
.50375 .24039 L
s
.5 .25754 m
.50375 .25754 L
s
.5 .2747 m
.50375 .2747 L
s
.5 .29185 m
.50375 .29185 L
s
.5 .32617 m
.50375 .32617 L
s
.5 .34332 m
.50375 .34332 L
s
.5 .36048 m
.50375 .36048 L
s
.5 .37763 m
.50375 .37763 L
s
.5 .41195 m
.50375 .41195 L
s
.5 .4291 m
.50375 .4291 L
s
.5 .44626 m
.50375 .44626 L
s
.5 .46342 m
.50375 .46342 L
s
.5 .49773 m
.50375 .49773 L
s
.5 .51488 m
.50375 .51488 L
s
.5 .53204 m
.50375 .53204 L
s
.5 .5492 m
.50375 .5492 L
s
.5 .03451 m
.50375 .03451 L
s
.5 .01736 m
.50375 .01736 L
s
.5 .0002 m
.50375 .0002 L
s
.5 .58351 m
.50375 .58351 L
s
.5 .60066 m
.50375 .60066 L
s
.5 .61782 m
.50375 .61782 L
s
.25 Mabswid
.5 0 m
.5 .61803 L
s
.025 w
.02381 .30901 Mdot
.0873 .30876 Mdot
.15079 .30954 Mdot
.21429 .30841 Mdot
.27778 .31719 Mdot
.34127 .37839 Mdot
.40476 .56366 Mdot
.46825 .51122 Mdot
.53175 .1068 Mdot
.59524 .05436 Mdot
.65873 .23963 Mdot
.72222 .30083 Mdot
.78571 .30961 Mdot
.84921 .30848 Mdot
.9127 .30926 Mdot
.97619 .30901 Mdot
.5 Mabswid
.02381 .30901 m
.02605 .30901 L
.02846 .30901 L
.03279 .30901 L
.03543 .30901 L
.03793 .30901 L
.04018 .30901 L
.04262 .30901 L
.045 .30901 L
.04762 .30901 L
.05036 .30901 L
.05293 .30901 L
.05751 .30901 L
.06009 .30901 L
.06244 .30901 L
.06504 .30901 L
.0678 .30901 L
.07041 .30901 L
.07277 .30901 L
.07556 .30901 L
.07858 .30901 L
.08143 .30901 L
.08405 .30901 L
.08881 .30901 L
.09151 .30901 L
.09403 .30901 L
.09912 .30901 L
.10198 .30901 L
.10458 .30901 L
.10939 .30901 L
.11181 .30901 L
.11449 .30901 L
.11677 .30901 L
.11922 .30901 L
.12366 .30901 L
.12618 .30901 L
.12894 .30901 L
.13163 .30901 L
.1345 .30901 L
.13719 .30901 L
.13964 .30901 L
.14443 .30901 L
.14927 .30901 L
.15172 .30901 L
.15441 .30901 L
.15916 .30902 L
.16361 .30902 L
.16885 .30902 L
.17364 .30903 L
.17916 .30904 L
Mistroke
.18441 .30905 L
.18945 .30906 L
.19402 .30908 L
.19929 .30911 L
.20425 .30915 L
.20886 .30919 L
.2131 .30925 L
.21775 .30932 L
.22265 .30942 L
.22763 .30956 L
.23216 .30972 L
.23738 .30995 L
.24233 .31024 L
.25112 .31095 L
.25609 .3115 L
.26068 .31213 L
.26532 .31291 L
.27033 .31394 L
.27938 .31644 L
.28462 .31834 L
.29014 .32078 L
.29999 .32652 L
.30957 .33419 L
.31978 .34517 L
.32941 .3587 L
.33816 .37396 L
.35738 .41813 L
.37819 .47997 L
.39771 .54157 L
.40806 .56989 L
.41376 .5827 L
.41916 .5924 L
.42164 .59592 L
.42427 .59897 L
.42564 .60025 L
.42715 .60142 L
.42853 .60225 L
.42983 .60282 L
.43101 .60316 L
.43227 .60332 L
.43334 .6033 L
.43451 .6031 L
.43577 .60269 L
.43648 .60236 L
.43712 .60201 L
.43956 .60014 L
.44074 .59893 L
.442 .59743 L
.44426 .59416 L
.44936 .58404 L
Mistroke
.45376 .57221 L
.45862 .5559 L
.46876 .51114 L
.47804 .45905 L
.49892 .31677 L
.51953 .17408 L
.52956 .11557 L
.53865 .07279 L
.54323 .05551 L
.54818 .04023 L
.55285 .02912 L
.55715 .02173 L
.55952 .01881 L
.56172 .01679 L
.56299 .01595 L
.56415 .01537 L
.56478 .01512 L
.56548 .01492 L
.56674 .01472 L
.56795 .01472 L
.56864 .0148 L
.56929 .01494 L
.5706 .01537 L
.57133 .0157 L
.572 .01606 L
.57437 .01778 L
.57686 .02027 L
.58187 .02726 L
.58642 .03568 L
.59662 .06027 L
.61499 .11617 L
.63581 .18094 L
.655 .22999 L
.67369 .26404 L
.68312 .27629 L
.69348 .28653 L
.70365 .29385 L
.71294 .29866 L
.71808 .30071 L
.72367 .30252 L
.73384 .30495 L
.73918 .30587 L
.7442 .30655 L
.75357 .30749 L
.75883 .30785 L
.76366 .30812 L
.76896 .30834 L
.77456 .30852 L
.77954 .30864 L
.78407 .30873 L
Mistroke
.7893 .30881 L
.79425 .30886 L
.79883 .3089 L
.80305 .30892 L
.80769 .30895 L
.8126 .30896 L
.81751 .30898 L
.82201 .30898 L
.8272 .30899 L
.83213 .309 L
.83666 .309 L
.84087 .309 L
.8455 .30901 L
.85043 .30901 L
.85303 .30901 L
.85578 .30901 L
.8607 .30901 L
.86345 .30901 L
.86601 .30901 L
.86899 .30901 L
.87183 .30901 L
.87426 .30901 L
.87692 .30901 L
.88156 .30901 L
.88419 .30901 L
.88698 .30901 L
.88942 .30901 L
.892 .30901 L
.89433 .30901 L
.89688 .30901 L
.90136 .30901 L
.90389 .30901 L
.90659 .30901 L
.91146 .30901 L
.91382 .30901 L
.91638 .30901 L
.91864 .30901 L
.92103 .30901 L
.92319 .30901 L
.92556 .30901 L
.92973 .30901 L
.93223 .30901 L
.93494 .30901 L
.93734 .30901 L
.93989 .30901 L
.94242 .30901 L
.94471 .30901 L
.94917 .30901 L
.95187 .30901 L
.95434 .30901 L
Mistroke
.95706 .30901 L
.95993 .30901 L
.96243 .30901 L
.96476 .30901 L
.96732 .30901 L
.97006 .30901 L
.97321 .30901 L
.97619 .30901 L
Mfstroke
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[61]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oS`3oool2
000008l0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0
oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo00l0oooo1000
001k0?ooo`00S`3oool2000000l0oooo00@000000?ooo`3oool0oooo0P00001j0?ooo`00S`3oool0
0`000000oooo0?ooo`0=0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0N03oool0
08l0oooo00<000000?ooo`3oool0303oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo
07L0oooo002?0?ooo`030000003oool0oooo00`0oooo00<000000?ooo`3oool01P3oool00`000000
oooo0?ooo`1f0?ooo`00S`3oool2000000d0oooo00<000000?ooo`3oool01P3oool00`000000oooo
0?ooo`1f0?ooo`00S`3oool00`000000oooo0?ooo`0;0?ooo`030000003oool0oooo00P0oooo00<0
00000?ooo`3oool0M@3oool008l0oooo00<000000?ooo`3oool02`3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo07D0oooo001e0?ooo`800000103oool2000000D0oooo00<000000?oo
o`3oool00P3oool2000000H0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0:0?oo
o`<00000M03oool007@0oooo00@000000?ooo`3oool000002P3oool01@000000oooo0?ooo`3oool0
00000080oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0:0?ooo`030000003oool0
oooo00T0oooo1@00001c0?ooo`00M03oool010000000oooo0?ooo`00000:0?ooo`030000003oool0
oooo00@0oooo00<000000?ooo`3oool00`3oool3000000T0oooo00<000000?ooo`3oool02@3oool7
00000780oooo001^0?ooo`@000000P3oool010000000oooo0?ooo`00000;0?ooo`040000003oool0
oooo0?ooo`8000001P3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo1`00
001b0?ooo`00M03oool010000000oooo0?ooo`0000080?ooo`040000003oool0oooo000000<0oooo
00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo00T0oooo
1`00001b0?ooo`00M@3oool2000000T0oooo100000030?ooo`<000001@3oool00`000000oooo0?oo
o`090?ooo`030000003oool0oooo00X0oooo1@00001c0?ooo`00S`3oool00`000000oooo0?ooo`08
0?ooo`030000003oool0oooo00`0oooo0`00001d0?ooo`00S`3oool2000000T0oooo00<000000?oo
o`3oool03`3oool00`000000oooo0?ooo`1a0?ooo`00S`3oool00`000000oooo0?ooo`080?ooo`03
0000003oool0oooo00l0oooo00<000000?ooo`3oool0L@3oool008l0oooo00<000000?ooo`3oool0
203oool00`000000oooo0?ooo`0?0?ooo`030000003oool0oooo0740oooo002?0?ooo`030000003o
ool0oooo00P0oooo00<000000?ooo`3oool0403oool00`000000oooo0?ooo`1`0?ooo`00S`3oool0
0`000000oooo0?ooo`070?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0L03oool0
08l0oooo0P0000080?ooo`030000003oool0oooo0140oooo00<000000?ooo`3oool0L03oool008l0
oooo00<000000?ooo`3oool01`3oool00`000000oooo0?ooo`0A0?ooo`030000003oool0oooo0700
oooo002?0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool04P3oool00`000000oooo
0?ooo`1_0?ooo`00S`3oool00`000000oooo0?ooo`050?ooo`<00000503oool00`000000oooo0?oo
o`1_0?ooo`00S`3oool00`000000oooo0?ooo`040?ooo`D000004`3oool00`000000oooo0?ooo`1_
0?ooo`00S`3oool2000000@0oooo1`00000C0?ooo`030000003oool0oooo06h0oooo002?0?ooo`03
0000003oool0oooo00<0oooo1`00000C0?ooo`030000003oool0oooo06h0oooo002?0?ooo`030000
003oool0oooo00<0oooo1`00000C0?ooo`030000003oool0oooo06h0oooo002?0?ooo`030000003o
ool0oooo00@0oooo1@00000E0?ooo`030000003oool0oooo06d0oooo002?0?ooo`030000003oool0
oooo00D0oooo0`00000F0?ooo`030000003oool0oooo06d0oooo002?0?ooo`8000001P3oool00`00
0000oooo0?ooo`0F0?ooo`030000003oool0oooo06d0oooo002?0?ooo`030000003oool0oooo00D0
oooo00<000000?ooo`3oool05`3oool00`000000oooo0?ooo`1/0?ooo`00S`3oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool0K03oool007/0oooo0P00
00040?ooo`800000103oool2000000H0oooo00<000000?ooo`3oool0103oool00`000000oooo0?oo
o`0H0?ooo`030000003oool0oooo06`0oooo001j0?ooo`040000003oool0oooo000000P0oooo00@0
00000?ooo`3oool000001@3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01T0oooo
00<000000?ooo`3oool0J`3oool007X0oooo00@000000?ooo`3oool000002`3oool00`000000oooo
0?ooo`030?ooo`<00000103oool00`000000oooo0?ooo`0I0?ooo`030000003oool0oooo06/0oooo
001d0?ooo`@000000P3oool010000000oooo0?ooo`0000090?ooo`8000001P3oool00`000000oooo
0?ooo`040?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool0J`3oool007X0oooo00@0
00000?ooo`3oool000002@3oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00@0oooo
00<000000?ooo`3oool06P3oool00`000000oooo0?ooo`1Z0?ooo`00N`3oool2000000X0oooo0`00
00050?ooo`030000003oool0oooo00@0oooo00<000000?ooo`3oool06P3oool00`000000oooo0?oo
o`1Z0?ooo`00S`3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo01X0oooo00<00000
0?ooo`3oool0JP3oool008l0oooo0P0000050?ooo`030000003oool0oooo01/0oooo00<000000?oo
o`3oool0J@3oool008l0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`0L0?ooo`03
0000003oool0oooo06T0oooo002?0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0
703oool00`000000oooo0?ooo`1Y0?ooo`00S`3oool00`000000oooo0?ooo`030?ooo`030000003o
ool0oooo01d0oooo00<000000?ooo`3oool0J03oool008l0oooo00<000000?ooo`3oool00`3oool0
0`000000oooo0?ooo`0M0?ooo`030000003oool0oooo06P0oooo002?0?ooo`800000103oool00`00
0000oooo0?ooo`0M0?ooo`030000003oool0oooo06P0oooo002?0?ooo`030000003oool0oooo00<0
oooo00<000000?ooo`3oool07P3oool00`000000oooo0?ooo`1W0?ooo`00S`3oool00`000000oooo
0?ooo`030?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool0I`3oool008l0oooo00<0
00000?ooo`3oool00`3oool00`000000oooo0?ooo`0N0?ooo`030000003oool0oooo06L0oooo002?
0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0803oool00`000000oooo0?ooo`1V
0?ooo`00S`3oool2000000<0oooo00<000000?ooo`3oool0803oool00`000000oooo0?ooo`1V0?oo
o`00S`3oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo0200oooo00<000000?ooo`3o
ool0IP3oool008l0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`0Q0?ooo`030000
003oool0oooo06D0oooo002?0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool08@3o
ool00`000000oooo0?ooo`1U0?ooo`00S`3oool00`000000oooo0?ooo`020?ooo`030000003oool0
oooo0240oooo00<000000?ooo`3oool0I@3oool008l0oooo0P0000030?ooo`030000003oool0oooo
0280oooo00<000000?ooo`3oool0I03oool008l0oooo00D000000?ooo`3oool0oooo0000000U0?oo
o`030000003oool0oooo06@0oooo001e0?ooo`800000103oool2000000<0oooo100000030?ooo`80
00001P3oool01@000000oooo0?ooo`3oool0000002H0oooo00<000000?ooo`3oool0H`3oool007@0
oooo00@000000?ooo`3oool00000203oool00`000000oooo0?ooo`030?ooo`040000003oool0oooo
000000D0oooo00D000000?ooo`3oool0oooo0000000V0?ooo`030000003oool0oooo06<0oooo001d
0?ooo`040000003oool0oooo000000T0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?oo
o`030?ooo`<0000000<0oooo0000003oool09@3oool00`000000oooo0?ooo`1S0?ooo`00KP3oool4
00000080oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`020?ooo`8000001P3o
ool01@000000oooo0?ooo`3oool0000002L0oooo00<000000?ooo`3oool0HP3oool007@0oooo00@0
00000?ooo`3oool00000203oool010000000oooo0?ooo`0000030?ooo`030000003oool0oooo00D0
oooo00D000000?ooo`3oool0oooo0000000W0?ooo`<00000HP3oool007D0oooo0P00000:0?ooo`80
0000103oool3000000D0oooo00D000000?ooo`3oool0oooo0000000V0?ooo`D00000H@3oool008l0
oooo00@000000?ooo`3oool000009P3oool700000600oooo002?0?ooo`80000000<0oooo0000003o
ool09@3oool700000600oooo002?0?ooo`040000003oool0oooo000002H0oooo1`00001P0?ooo`00
S`3oool010000000oooo0?ooo`00000W0?ooo`D00000H@3oool008l0oooo00@000000?ooo`3oool0
0000:03oool3000000030?ooo`000000oooo05l0oooo002?0?ooo`040000003oool0oooo000002`0
oooo00<000000?ooo`3oool0GP3oool008l0oooo0`00000^0?ooo`030000003oool0oooo05d0oooo
002?0?ooo`030000003oool0000002h0oooo00<000000?ooo`3oool0G@3oool008l0oooo00<00000
0?ooo`000000;`3oool00`000000oooo0?ooo`1L0?ooo`009@3oool300000340oooo1000000b0?oo
o`030000003oool0000002l0oooo00<000000?ooo`3oool01000000d0?ooo`<000008@3oool002H0
oooo00<000000?ooo`3oool0<03oool00`000000oooo0?ooo`0c0?ooo`030000003oool000000300
oooo00<000000?ooo`000000>03oool00`000000oooo0?ooo`0P0?ooo`008`3oool500000380oooo
00<000000?ooo`3oool0<P3oool300000300oooo00@000000?ooo`3oool00000=03oool500000240
oooo000M0?ooo`@000000P3oool010000000oooo0?ooo`00000/0?ooo`@00000103oool00`000000
oooo0?ooo`0a0?ooo`030000003oool000000340oooo00@000000?ooo`3oool00000<`3oool01000
0000oooo0?ooo`00000R0?ooo`00903oool00`000000oooo0000000b0?ooo`040000003oool0oooo
00000380oooo0P00000c0?ooo`040000003oool0oooo000003<0oooo00<000000?ooo`0000008P3o
ool002D0oooo0P00000c0?ooo`800000<`3oool2000003@0oooo0P00000e0?ooo`8000008P3oool0
08l0oooo0P00000e0?ooo`030000003oool0oooo05L0oooo002?0?ooo`800000=P3oool2000000D0
oooo0`00001?0?ooo`00S`3oool2000003P0oooo0P0000020?ooo`D00000CP3oool000H0oooo0`00
000?0?ooo`<000003`3oool3000000l0oooo0`00001@0?ooo`800000>P3oool8000000d0oooo0`00
000?0?ooo`<000003`3oool3000000l0oooo0`0000070?ooo`001@3oool5000000d0oooo1@00000=
0?ooo`D000003@3oool5000004l0oooo0P00000k0?ooo`L00000303oool5000000d0oooo1@00000=
0?ooo`D000003@3oool5000000H0oooo00040?ooo`L000002`3oool7000000/0oooo1`00000;0?oo
o`L000003@3oool3000003h0oooo00<000000?ooo`3oool0>P3oool;000000L0oooo1`00000;0?oo
o`L000002`3oool7000000/0oooo1`0000050?ooo`00o`00000P00000040oooo00040?ooo`L00000
2@3oool00`000000oooo00000006000000D0oooo00<000000?ooo`3oool00`3oool7000000030?oo
o`000000oooo00P0oooo1`0000060?ooo``000001@3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool02P3oool00`000000oooo0?ooo`0;0?ooo`030000
003oool0oooo00/0oooo00<000000?ooo`3oool01@3oool300000080oooo00<000000?ooo`3oool0
203oool7000000T0oooo00<000000?ooo`0000001P0000050?ooo`030000003oool0oooo00<0oooo
1`0000000`3oool000000?ooo`080?ooo`L000001@3oool000D0oooo1@00000=0?ooo`D000001P3o
ool00`000000oooo0?ooo`040?ooo`D000003@3oool5000000`0oooo1`0000050?ooo`030000003o
ool0oooo03@0oooo00<000000?ooo`3oool0=03oool00`000000oooo0?ooo`0F0?ooo`D000003@3o
ool5000000H0oooo00<000000?ooo`3oool0103oool5000000d0oooo1@0000060?ooo`001P3oool3
000000l0oooo0`00000?0?ooo`<000003`3oool3000000d0oooo2000000k0?ooo`030000003oool0
oooo04h0oooo0`00000?0?ooo`<000003`3oool3000000l0oooo0`0000070?ooo`00C@3oool50000
0080oooo0P00000i0?ooo`030000003oool0oooo08h0oooo001>0?ooo`<000001@3oool00`000000
oooo0?ooo`0f0?ooo`800000S`3oool005L0oooo00<000000?ooo`3oool0=03oool200000900oooo
001H0?ooo`030000003oool0oooo03<0oooo0P00002@0?ooo`00F@3oool00`000000oooo0?ooo`0b
0?ooo`800000T03oool005X0oooo00<000000?ooo`3oool0<@3oool200000900oooo001K0?ooo`03
0000003oool0oooo0300oooo0`00002?0?ooo`00G03oool00`000000oooo0?ooo`0_0?ooo`800000
T03oool005`0oooo00<000000?ooo`3oool0;`3oool200000900oooo001M0?ooo`030000003oool0
oooo02d0oooo00<000000?ooo`000000T03oool005h0oooo00<000000?ooo`3oool0;03oool00`00
0000oooo0000002@0?ooo`00G`3oool00`000000oooo0?ooo`0[0?ooo`040000003oool000000000
08l0oooo001O0?ooo`030000003oool0oooo02/0oooo00<000000?ooo`000000T03oool00600oooo
0`00000Z0?ooo`030000003oool000000900oooo001O0?ooo`D00000:@3oool00`000000oooo0000
002@0?ooo`00GP3oool7000002P0oooo00<000000?ooo`000000T03oool005h0oooo1`00000W0?oo
o`030000003oool0oooo00800000S`3oool005h0oooo1`00000W0?ooo`040000003oool0oooo0000
0900oooo001O0?ooo`D00000:03oool010000000oooo0?ooo`00002@0?ooo`00H03oool400000140
oooo0P0000040?ooo`8000000`3oool4000000<0oooo0P0000030?ooo`040000003oool0oooo0000
0900oooo001S0?ooo`030000003oool0oooo00h0oooo00@000000?ooo`3oool00000203oool00`00
0000oooo0?ooo`030?ooo`040000003oool0oooo00000080oooo00@000000?ooo`3oool00000T03o
ool006<0oooo00<000000?ooo`3oool03P3oool010000000oooo0?ooo`0000090?ooo`030000003o
ool0oooo00D0oooo00@000000?ooo`3oool000000P3oool3000008h0oooo001T0?ooo`030000003o
ool0oooo00d0oooo00@000000?ooo`3oool000002P3oool00`000000oooo0?ooo`020?ooo`800000
0P3oool01@000000oooo0?ooo`3oool000000900oooo001T0?ooo`030000003oool0oooo00d0oooo
00@000000?ooo`3oool00000203oool010000000oooo0?ooo`0000030?ooo`050000003oool0oooo
0?ooo`0000000`3oool00`000000oooo0?ooo`2>0?ooo`00I@3oool00`000000oooo0?ooo`0=0?oo
o`8000002P3oool2000000@0oooo0`0000000`3oool000000?ooo`020?ooo`030000003oool0oooo
08h0oooo001U0?ooo`030000003oool0oooo02<0oooo00@000000?ooo`3oool0oooo0P00002?0?oo
o`00IP3oool00`000000oooo0?ooo`0R0?ooo`050000003oool0oooo0?ooo`000000T03oool006H0
oooo00<000000?ooo`3oool08P3oool01@000000oooo0?ooo`3oool000000900oooo001V0?ooo`03
0000003oool0oooo0280oooo00D000000?ooo`3oool0oooo0000002@0?ooo`00I`3oool00`000000
oooo0?ooo`0P0?ooo`030000003oool0oooo0080oooo00<000000?ooo`3oool0SP3oool006L0oooo
00<000000?ooo`3oool0803oool00`000000oooo0?ooo`020?ooo`800000S`3oool006L0oooo00<0
00000?ooo`3oool0803oool00`000000oooo0?ooo`020?ooo`030000003oool0oooo08h0oooo001X
0?ooo`030000003oool0oooo01l0oooo00<000000?ooo`3oool00P3oool00`000000oooo0?ooo`2>
0?ooo`00J03oool00`000000oooo0?ooo`0O0?ooo`030000003oool0oooo0080oooo00<000000?oo
o`3oool0SP3oool006P0oooo00<000000?ooo`3oool07`3oool00`000000oooo0?ooo`020?ooo`03
0000003oool0oooo08h0oooo001Y0?ooo`030000003oool0oooo01h0oooo00<000000?ooo`3oool0
0P3oool2000008l0oooo001Y0?ooo`030000003oool0oooo01d0oooo00<000000?ooo`3oool00`3o
ool00`000000oooo0?ooo`2>0?ooo`00J@3oool00`000000oooo0?ooo`0M0?ooo`030000003oool0
oooo00<0oooo00<000000?ooo`3oool0SP3oool006X0oooo00<000000?ooo`3oool0703oool00`00
0000oooo0?ooo`030?ooo`030000003oool0oooo08h0oooo001Z0?ooo`030000003oool0oooo01`0
oooo00<000000?ooo`3oool00`3oool00`000000oooo0?ooo`2>0?ooo`00JP3oool00`000000oooo
0?ooo`0L0?ooo`030000003oool0oooo00<0oooo0P00002?0?ooo`00J`3oool00`000000oooo0?oo
o`0K0?ooo`030000003oool0oooo00<0oooo00<000000?ooo`3oool0SP3oool006/0oooo00<00000
0?ooo`3oool06P3oool00`000000oooo0?ooo`040?ooo`030000003oool0oooo08h0oooo001[0?oo
o`030000003oool0oooo00d0oooo0P0000040?ooo`800000103oool2000000H0oooo00<000000?oo
o`3oool0SP3oool006`0oooo00<000000?ooo`3oool02`3oool010000000oooo0?ooo`0000080?oo
o`040000003oool00000000000D0oooo00<000000?ooo`3oool0SP3oool006`0oooo00<000000?oo
o`3oool02`3oool010000000oooo0?ooo`00000:0?ooo`8000001@3oool3000008h0oooo001/0?oo
o`030000003oool0oooo00/0oooo00@000000?ooo`3oool000002@3oool2000000H0oooo00<00000
0?ooo`3oool0SP3oool006d0oooo00<000000?ooo`3oool02P3oool010000000oooo0?ooo`000009
0?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0SP3oool006d0oooo00<000000?oo
o`3oool02`3oool2000000X0oooo0`0000050?ooo`030000003oool0oooo08h0oooo001]0?ooo`03
0000003oool0oooo01L0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`2>0?ooo`00
KP3oool00`000000oooo0?ooo`0F0?ooo`030000003oool0oooo00D0oooo0P00002?0?ooo`00KP3o
ool00`000000oooo0?ooo`0D0?ooo`<000001`3oool00`000000oooo0?ooo`2>0?ooo`00KP3oool0
0`000000oooo0?ooo`0C0?ooo`D000001P3oool00`000000oooo0?ooo`2>0?ooo`00K`3oool00`00
0000oooo0?ooo`0A0?ooo`L000001@3oool00`000000oooo0?ooo`2>0?ooo`00K`3oool00`000000
oooo0?ooo`0A0?ooo`L000001@3oool00`000000oooo0?ooo`2>0?ooo`00K`3oool00`000000oooo
0?ooo`0A0?ooo`L000001@3oool2000008l0oooo001`0?ooo`030000003oool0oooo0140oooo1@00
00060?ooo`030000003oool0oooo08h0oooo001`0?ooo`030000003oool0oooo0180oooo0`000007
0?ooo`030000003oool0oooo08h0oooo001`0?ooo`030000003oool0oooo0180oooo00<000000?oo
o`3oool01`3oool00`000000oooo0?ooo`2>0?ooo`00L@3oool00`000000oooo0?ooo`0A0?ooo`03
0000003oool0oooo00L0oooo00<000000?ooo`3oool0SP3oool00740oooo00<000000?ooo`3oool0
4@3oool00`000000oooo0?ooo`070?ooo`800000S`3oool00740oooo00<000000?ooo`3oool0403o
ool00`000000oooo0?ooo`080?ooo`030000003oool0oooo08h0oooo001b0?ooo`030000003oool0
oooo00l0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`2>0?ooo`00LP3oool00`00
0000oooo0?ooo`0?0?ooo`030000003oool0oooo00P0oooo00<000000?ooo`3oool0SP3oool00780
oooo00<000000?ooo`3oool03`3oool00`000000oooo0?ooo`080?ooo`030000003oool0oooo08h0
oooo001c0?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3oool02@3oool2000008l0oooo
001c0?ooo`<000003@3oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo08h0oooo001b
0?ooo`D00000303oool00`000000oooo0?ooo`090?ooo`030000003oool0oooo08h0oooo001a0?oo
o`L000000`3oool2000000D0oooo0P0000030?ooo`8000001P3oool00`000000oooo0?ooo`2>0?oo
o`00L@3oool7000000X0oooo00D000000?ooo`3oool0oooo000000020?ooo`030000003oool0oooo
00<0oooo00<000000?ooo`3oool0SP3oool00740oooo1`00000:0?ooo`030000003oool0oooo00@0
oooo00<000000?ooo`3oool00`3oool3000008h0oooo001b0?ooo`H000002P3oool2000000<0oooo
0P0000060?ooo`030000003oool0oooo08h0oooo001c0?ooo`<0000000<0oooo0000003oool01`3o
ool010000000oooo0000000000030?ooo`030000003oool0oooo00D0oooo00<000000?ooo`3oool0
SP3oool007D0oooo0P0000090?ooo`@000000`3oool3000000D0oooo00<000000?ooo`3oool0SP3o
ool007H0oooo00<000000?ooo`3oool0203oool00`000000oooo0?ooo`0;0?ooo`030000003oool0
oooo08h0oooo001g0?ooo`030000003oool0oooo00L0oooo00<000000?ooo`3oool02`3oool20000
08l0oooo001g0?ooo`030000003oool0oooo00H0oooo00<000000?ooo`3oool0303oool00`000000
oooo0?ooo`2>0?ooo`00N03oool00`000000oooo0?ooo`050?ooo`030000003oool0oooo00`0oooo
00<000000?ooo`3oool0SP3oool007T0oooo00<000000?ooo`3oool00`3oool00`000000oooo0?oo
o`0=0?ooo`030000003oool0oooo08h0oooo001i0?ooo`8000000`3oool00`000000oooo0?ooo`0>
0?ooo`030000003oool0oooo08h0oooo001j0?ooo`@000004@3oool2000008l0oooo002?0?ooo`03
0000003oool0oooo08h0oooo002?0?ooo`030000003oool0oooo08h0oooo002?0?ooo`030000003o
ool0oooo08h0oooo002?0?ooo`800000S`3oool00001\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-5.26302, -0.90059, \
0.0366761, 0.0101799}}]
}, Open  ]],

Cell["\<\
With the assumption of periodicity, the approximation is \
significantly improved. The accuracy of the periodic pseudospectral \
approximations is sufficiently high to justify, in some cases, using a larger \
computational domain to simulate periodicity, say for a pulse like the above \
example. Despite the great accuracy of these approximations, they are not \
without pitfalls: one of the worst is probably aliasing error, whereby an \
oscillatory function component with too great a frequency can be \
misapproximated or disappear entirely.\
\>", "Text",
  CellTags->{"b:7.1.2", "ndsg:2.0.1.2"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Accuracy and Convergence of Finite Difference Approximations", \
"Subsubsection",
  CellTags->{"b:7.1.3", "ndsg:2.0.1.3", "PDE:1.2.5"}],

Cell[TextData[{
  "When using finite differences, it is important to keep in mind that the \
truncation error, or the asymptotic approximation error induced by cutting \
off the Taylor series approximation, is not the only source of error. There \
are two other sources of error in applying finite difference formulas; \
condition error and roundoff error [",
  ButtonBox["GMW81",
    ButtonData:>{"References.nb", "GMW81"},
    ButtonStyle->"Hyperlink"],
  "]. Roundoff error comes from roundoff in the arithmetic computations \
required.  Condition error comes from magnification of any errors in the \
function values, typically from the division by a power of the step size, and \
so grows with decreasing step size. This means that in practice, even though \
the truncation error approaches zero as h does, the actual error will start \
growing beyond some point. ",
  " ",
  "The figures below demonstrate the typical behavior as h becomes small for \
a smooth function."
}], "Text"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.293651 0.263645 0.576668 0.0407324 [
[.23364 .3009 -11 -9.8125 ]
[.23364 .3009 11 0 ]
[.49728 .3009 -14 -9.8125 ]
[.49728 .3009 14 0 ]
[.76093 .3009 -17 -9.8125 ]
[.76093 .3009 17 0 ]
[-0.0125 .00641 -51.6875 -6.15625 ]
[-0.0125 .00641 0 6.15625 ]
[-0.0125 .12861 -51.6875 -6.15625 ]
[-0.0125 .12861 0 6.15625 ]
[-0.0125 .25081 -47.4375 -6.15625 ]
[-0.0125 .25081 0 6.15625 ]
[-0.0125 .37301 -46 -4.90625 ]
[-0.0125 .37301 0 4.90625 ]
[-0.0125 .4952 -28 -4.90625 ]
[-0.0125 .4952 0 4.90625 ]
[-0.0125 .6174 -16 -4.90625 ]
[-0.0125 .6174 0 4.90625 ]
[ -0.0005 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.23364 .3134 m
.23364 .31965 L
s
gsave
.23364 .3009 -72 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(100) show
81.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.49728 .3134 m
.49728 .31965 L
s
gsave
.49728 .3009 -75 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1000) show
87.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.76093 .3134 m
.76093 .31965 L
s
gsave
.76093 .3009 -78 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10000) show
93.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.04936 .3134 m
.04936 .31715 L
s
.09578 .3134 m
.09578 .31715 L
s
.12872 .3134 m
.12872 .31715 L
s
.15427 .3134 m
.15427 .31715 L
s
.17515 .3134 m
.17515 .31715 L
s
.1928 .3134 m
.1928 .31715 L
s
.20809 .3134 m
.20809 .31715 L
s
.22158 .3134 m
.22158 .31715 L
s
.313 .3134 m
.313 .31715 L
s
.35943 .3134 m
.35943 .31715 L
s
.39237 .3134 m
.39237 .31715 L
s
.41792 .3134 m
.41792 .31715 L
s
.4388 .3134 m
.4388 .31715 L
s
.45645 .3134 m
.45645 .31715 L
s
.47173 .3134 m
.47173 .31715 L
s
.48522 .3134 m
.48522 .31715 L
s
.57665 .3134 m
.57665 .31715 L
s
.62308 .3134 m
.62308 .31715 L
s
.65601 .3134 m
.65601 .31715 L
s
.68156 .3134 m
.68156 .31715 L
s
.70244 .3134 m
.70244 .31715 L
s
.72009 .3134 m
.72009 .31715 L
s
.73538 .3134 m
.73538 .31715 L
s
.74887 .3134 m
.74887 .31715 L
s
.84029 .3134 m
.84029 .31715 L
s
.88672 .3134 m
.88672 .31715 L
s
.91966 .3134 m
.91966 .31715 L
s
.94521 .3134 m
.94521 .31715 L
s
.96609 .3134 m
.96609 .31715 L
s
.98374 .3134 m
.98374 .31715 L
s
.99902 .3134 m
.99902 .31715 L
s
.25 Mabswid
0 .3134 m
1 .3134 L
s
0 .00641 m
.00625 .00641 L
s
gsave
-0.0125 .00641 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(14) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .12861 m
.00625 .12861 L
s
gsave
-0.0125 .12861 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(11) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .25081 m
.00625 .25081 L
s
gsave
-0.0125 .25081 -108.438 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(8) show
105.125 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .37301 m
.00625 .37301 L
s
gsave
-0.0125 .37301 -107 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.00001) show
105.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .4952 m
.00625 .4952 L
s
gsave
-0.0125 .4952 -89 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.01) show
87.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .6174 m
.00625 .6174 L
s
gsave
-0.0125 .6174 -77 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
75.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .08988 m
.00375 .08988 L
s
0 .10207 m
.00375 .10207 L
s
0 .10921 m
.00375 .10921 L
s
0 .11429 m
.00375 .11429 L
s
0 .11823 m
.00375 .11823 L
s
0 .12145 m
.00375 .12145 L
s
0 .12417 m
.00375 .12417 L
s
0 .12653 m
.00375 .12653 L
s
0 .21208 m
.00375 .21208 L
s
0 .22426 m
.00375 .22426 L
s
0 .23141 m
.00375 .23141 L
s
0 .23649 m
.00375 .23649 L
s
0 .24043 m
.00375 .24043 L
s
0 .24365 m
.00375 .24365 L
s
0 .24637 m
.00375 .24637 L
s
0 .24873 m
.00375 .24873 L
s
0 .33428 m
.00375 .33428 L
s
0 .34646 m
.00375 .34646 L
s
0 .35361 m
.00375 .35361 L
s
0 .35868 m
.00375 .35868 L
s
0 .36262 m
.00375 .36262 L
s
0 .36584 m
.00375 .36584 L
s
0 .36857 m
.00375 .36857 L
s
0 .37093 m
.00375 .37093 L
s
0 .45648 m
.00375 .45648 L
s
0 .46866 m
.00375 .46866 L
s
0 .4758 m
.00375 .4758 L
s
0 .48088 m
.00375 .48088 L
s
0 .48482 m
.00375 .48482 L
s
0 .48804 m
.00375 .48804 L
s
0 .49076 m
.00375 .49076 L
s
0 .49312 m
.00375 .49312 L
s
0 .57867 m
.00375 .57867 L
s
0 .59086 m
.00375 .59086 L
s
0 .598 m
.00375 .598 L
s
0 .60308 m
.00375 .60308 L
s
0 .60702 m
.00375 .60702 L
s
0 .61024 m
.00375 .61024 L
s
0 .61296 m
.00375 .61296 L
s
0 .61532 m
.00375 .61532 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
1 0 0 r
.015 w
.02381 .60082 Mdot
.04936 .60063 Mdot
.06027 .59915 Mdot
.0794 .59514 Mdot
.10317 .58832 Mdot
.12285 .58159 Mdot
.14473 .57558 Mdot
.16309 .57106 Mdot
.18254 .565 Mdot
.20222 .55899 Mdot
.22158 .55333 Mdot
.24245 .54693 Mdot
.2619 .54079 Mdot
.28158 .53489 Mdot
.30221 .52855 Mdot
.32182 .5225 Mdot
.34127 .51657 Mdot
.36095 .51047 Mdot
.38094 .50431 Mdot
.40065 .49824 Mdot
.42063 .49206 Mdot
.44031 .48598 Mdot
.46031 .4798 Mdot
.48028 .47364 Mdot
.5 .46754 Mdot
.51986 .4614 Mdot
.53967 .45529 Mdot
.55951 .44916 Mdot
.57937 .44302 Mdot
.59923 .43688 Mdot
.61904 .43076 Mdot
.63888 .42463 Mdot
.65873 .4185 Mdot
.67855 .41238 Mdot
.6984 .40624 Mdot
.71824 .40011 Mdot
.7381 .39398 Mdot
.75794 .38784 Mdot
.77779 .38171 Mdot
.79763 .37558 Mdot
.81746 .36945 Mdot
.8373 .36332 Mdot
.85714 .35719 Mdot
.87698 .35106 Mdot
.89683 .34493 Mdot
.91667 .3388 Mdot
.93651 .33267 Mdot
.95635 .32654 Mdot
.97619 .32041 Mdot
0 1 0 r
.02381 .58786 Mdot
.04936 .58737 Mdot
.06027 .58489 Mdot
.0794 .57831 Mdot
.10317 .56719 Mdot
.12285 .55619 Mdot
.14473 .54262 Mdot
.16309 .53229 Mdot
.18254 .522 Mdot
.20222 .51009 Mdot
.22158 .49824 Mdot
.24245 .48589 Mdot
.2619 .47395 Mdot
.28158 .4618 Mdot
.30221 .44926 Mdot
.32182 .43721 Mdot
.34127 .42519 Mdot
.36095 .41312 Mdot
.38094 .40075 Mdot
.40065 .38859 Mdot
.42063 .37628 Mdot
.44031 .36412 Mdot
.46031 .35178 Mdot
.48028 .33944 Mdot
.5 .32725 Mdot
.51986 .31498 Mdot
.53967 .30274 Mdot
.55951 .29048 Mdot
.57937 .27821 Mdot
.59923 .26593 Mdot
.61904 .2537 Mdot
.63888 .24143 Mdot
.65873 .22916 Mdot
.67855 .21692 Mdot
.6984 .20465 Mdot
.71824 .1924 Mdot
.7381 .18015 Mdot
.75794 .16799 Mdot
.77779 .15586 Mdot
.79763 .14397 Mdot
.81746 .13272 Mdot
.8373 .12242 Mdot
.85714 .11427 Mdot
.87698 .10907 Mdot
.89683 .1079 Mdot
.91667 .1081 Mdot
.93651 .11122 Mdot
.95635 .11425 Mdot
.97619 .11677 Mdot
0 0 1 r
.02381 .57707 Mdot
.04936 .57764 Mdot
.06027 .57452 Mdot
.0794 .56598 Mdot
.10317 .55123 Mdot
.12285 .53644 Mdot
.14473 .51799 Mdot
.16309 .50128 Mdot
.18254 .48543 Mdot
.20222 .46888 Mdot
.22158 .45117 Mdot
.24245 .43251 Mdot
.2619 .41478 Mdot
.28158 .39688 Mdot
.30221 .37782 Mdot
.32182 .3597 Mdot
.34127 .3418 Mdot
.36095 .32371 Mdot
.38094 .30523 Mdot
.40065 .28699 Mdot
.42063 .26849 Mdot
.44031 .25024 Mdot
.46031 .23173 Mdot
.48028 .21323 Mdot
.5 .19496 Mdot
.51986 .17657 Mdot
.53967 .15825 Mdot
.55951 .13992 Mdot
.57937 .12215 Mdot
.59923 .10524 Mdot
.61904 .09218 Mdot
.63888 .08328 Mdot
.65873 .08324 Mdot
.67855 .08435 Mdot
.6984 .09002 Mdot
.71824 .08872 Mdot
.7381 .09409 Mdot
.75794 .09755 Mdot
.77779 .10294 Mdot
.79763 .10427 Mdot
.81746 .10665 Mdot
.8373 .10986 Mdot
.85714 .113 Mdot
.87698 .11722 Mdot
.89683 .11997 Mdot
.91667 .12254 Mdot
.93651 .12528 Mdot
.95635 .13013 Mdot
.97619 .13225 Mdot
1 0 1 r
.02381 .57009 Mdot
.04936 .57034 Mdot
.06027 .56639 Mdot
.0794 .55634 Mdot
.10317 .53851 Mdot
.12285 .52034 Mdot
.14473 .49739 Mdot
.16309 .47642 Mdot
.18254 .45319 Mdot
.20222 .43227 Mdot
.22158 .40968 Mdot
.24245 .38388 Mdot
.2619 .3612 Mdot
.28158 .33684 Mdot
.30221 .31202 Mdot
.32182 .288 Mdot
.34127 .26409 Mdot
.36095 .23987 Mdot
.38094 .21521 Mdot
.40065 .19092 Mdot
.42063 .16626 Mdot
.44031 .14204 Mdot
.46031 .11768 Mdot
.48028 .09485 Mdot
.5 .07772 Mdot
.51986 .07338 Mdot
.53967 .07081 Mdot
.55951 .07338 Mdot
.57937 .0795 Mdot
.59923 .08421 Mdot
.61904 .08295 Mdot
.63888 .08756 Mdot
.65873 .09135 Mdot
.67855 .08805 Mdot
.6984 .09596 Mdot
.71824 .09967 Mdot
.7381 .10334 Mdot
.75794 .10772 Mdot
.77779 .11637 Mdot
.79763 .11855 Mdot
.81746 .11566 Mdot
.8373 .11984 Mdot
.85714 .11804 Mdot
.87698 .12463 Mdot
.89683 .12886 Mdot
.91667 .13259 Mdot
.93651 .13021 Mdot
.95635 .13722 Mdot
.97619 .14108 Mdot
0 0 0 r
.02381 .57368 Mdot
.04936 .54539 Mdot
.06027 .52869 Mdot
.0794 .49069 Mdot
.10317 .42161 Mdot
.12285 .3381 Mdot
.14473 .20469 Mdot
.16309 .04716 Mdot
.18254 .01544 Mdot
.20222 .01638 Mdot
.22158 .01472 Mdot
.24245 .02126 Mdot
.2619 .0297 Mdot
.28158 .0326 Mdot
.30221 .03898 Mdot
.32182 .03758 Mdot
.34127 .04996 Mdot
.36095 .04604 Mdot
.38094 .0694 Mdot
.40065 .04065 Mdot
.42063 .06475 Mdot
.44031 .05831 Mdot
.46031 .07947 Mdot
.48028 .08711 Mdot
.5 .08394 Mdot
.51986 .06495 Mdot
.53967 .09455 Mdot
.55951 .08223 Mdot
.57937 .09303 Mdot
.59923 .08236 Mdot
.61904 .11172 Mdot
.63888 .09676 Mdot
.65873 .10484 Mdot
.67855 .12688 Mdot
.6984 .12621 Mdot
.71824 .11358 Mdot
.7381 .1158 Mdot
.75794 .13582 Mdot
.77779 .13743 Mdot
.79763 .14667 Mdot
.81746 .13272 Mdot
.8373 .1545 Mdot
.85714 .15453 Mdot
.87698 .15286 Mdot
.89683 .14739 Mdot
.91667 .15887 Mdot
.93651 .16789 Mdot
.95635 .16761 Mdot
.97619 .16052 Mdot
.5 .5 .5 r
.02381 .60332 Mdot
.04936 .59227 Mdot
.06027 .58672 Mdot
.0794 .57234 Mdot
.10317 .54806 Mdot
.12285 .519 Mdot
.14473 .47224 Mdot
.16309 .41707 Mdot
.18254 .33739 Mdot
.20222 .22759 Mdot
.22158 .07825 Mdot
.24245 .04551 Mdot
.2619 .10182 Mdot
.28158 .09222 Mdot
.30221 .08655 Mdot
.32182 .08231 Mdot
.34127 .12348 Mdot
.36095 .12002 Mdot
.38094 .14031 Mdot
.40065 .13902 Mdot
.42063 .14632 Mdot
.44031 .13873 Mdot
.46031 .177 Mdot
.48028 .17383 Mdot
.5 .16317 Mdot
.51986 .15996 Mdot
.53967 .16018 Mdot
.55951 .17994 Mdot
.57937 .20926 Mdot
.59923 .16285 Mdot
.61904 .19572 Mdot
.63888 .20974 Mdot
.65873 .20302 Mdot
.67855 .23237 Mdot
.6984 .23536 Mdot
.71824 .23738 Mdot
.7381 .24347 Mdot
.75794 .2388 Mdot
.77779 .24013 Mdot
.79763 .27487 Mdot
.81746 .26774 Mdot
.8373 .27885 Mdot
.85714 .30323 Mdot
.87698 .2847 Mdot
.89683 .27299 Mdot
.91667 .29928 Mdot
.93651 .28457 Mdot
.95635 .2992 Mdot
.97619 .30105 Mdot
1 0 0 r
.5 Mabswid
.02381 .60082 m
.04936 .60063 L
.06027 .59915 L
.0794 .59514 L
.10317 .58832 L
.12285 .58159 L
.14473 .57558 L
.16309 .57106 L
.18254 .565 L
.20222 .55899 L
.22158 .55333 L
.24245 .54693 L
.2619 .54079 L
.28158 .53489 L
.30221 .52855 L
.32182 .5225 L
.34127 .51657 L
.36095 .51047 L
.38094 .50431 L
.40065 .49824 L
.42063 .49206 L
.44031 .48598 L
.46031 .4798 L
.48028 .47364 L
.5 .46754 L
.51986 .4614 L
.53967 .45529 L
.55951 .44916 L
.57937 .44302 L
.59923 .43688 L
.61904 .43076 L
.63888 .42463 L
.65873 .4185 L
.67855 .41238 L
.6984 .40624 L
.71824 .40011 L
.7381 .39398 L
.75794 .38784 L
.77779 .38171 L
.79763 .37558 L
.81746 .36945 L
.8373 .36332 L
.85714 .35719 L
.87698 .35106 L
.89683 .34493 L
.91667 .3388 L
.93651 .33267 L
.95635 .32654 L
.97619 .32041 L
s
0 1 0 r
.02381 .58786 m
.04936 .58737 L
.06027 .58489 L
.0794 .57831 L
.10317 .56719 L
.12285 .55619 L
.14473 .54262 L
.16309 .53229 L
.18254 .522 L
.20222 .51009 L
.22158 .49824 L
.24245 .48589 L
.2619 .47395 L
.28158 .4618 L
.30221 .44926 L
.32182 .43721 L
.34127 .42519 L
.36095 .41312 L
.38094 .40075 L
.40065 .38859 L
.42063 .37628 L
.44031 .36412 L
.46031 .35178 L
.48028 .33944 L
.5 .32725 L
.51986 .31498 L
.53967 .30274 L
.55951 .29048 L
.57937 .27821 L
.59923 .26593 L
.61904 .2537 L
.63888 .24143 L
.65873 .22916 L
.67855 .21692 L
.6984 .20465 L
.71824 .1924 L
.7381 .18015 L
.75794 .16799 L
.77779 .15586 L
.79763 .14397 L
.81746 .13272 L
.8373 .12242 L
.85714 .11427 L
.87698 .10907 L
.89683 .1079 L
.91667 .1081 L
.93651 .11122 L
.95635 .11425 L
.97619 .11677 L
s
0 0 1 r
.02381 .57707 m
.04936 .57764 L
.06027 .57452 L
.0794 .56598 L
.10317 .55123 L
.12285 .53644 L
.14473 .51799 L
.16309 .50128 L
.18254 .48543 L
.20222 .46888 L
.22158 .45117 L
.24245 .43251 L
.2619 .41478 L
.28158 .39688 L
.30221 .37782 L
.32182 .3597 L
.34127 .3418 L
.36095 .32371 L
.38094 .30523 L
.40065 .28699 L
.42063 .26849 L
.44031 .25024 L
.46031 .23173 L
.48028 .21323 L
.5 .19496 L
.51986 .17657 L
.53967 .15825 L
.55951 .13992 L
.57937 .12215 L
.59923 .10524 L
.61904 .09218 L
.63888 .08328 L
.65873 .08324 L
.67855 .08435 L
.6984 .09002 L
.71824 .08872 L
.7381 .09409 L
.75794 .09755 L
.77779 .10294 L
.79763 .10427 L
.81746 .10665 L
.8373 .10986 L
.85714 .113 L
.87698 .11722 L
.89683 .11997 L
.91667 .12254 L
.93651 .12528 L
.95635 .13013 L
.97619 .13225 L
s
1 0 1 r
.02381 .57009 m
.04936 .57034 L
.06027 .56639 L
.0794 .55634 L
.10317 .53851 L
.12285 .52034 L
.14473 .49739 L
.16309 .47642 L
.18254 .45319 L
.20222 .43227 L
.22158 .40968 L
.24245 .38388 L
.2619 .3612 L
.28158 .33684 L
.30221 .31202 L
.32182 .288 L
.34127 .26409 L
.36095 .23987 L
.38094 .21521 L
.40065 .19092 L
.42063 .16626 L
.44031 .14204 L
.46031 .11768 L
.48028 .09485 L
.5 .07772 L
.51986 .07338 L
.53967 .07081 L
.55951 .07338 L
.57937 .0795 L
.59923 .08421 L
.61904 .08295 L
.63888 .08756 L
.65873 .09135 L
.67855 .08805 L
.6984 .09596 L
.71824 .09967 L
.7381 .10334 L
.75794 .10772 L
.77779 .11637 L
.79763 .11855 L
.81746 .11566 L
.8373 .11984 L
.85714 .11804 L
.87698 .12463 L
.89683 .12886 L
.91667 .13259 L
.93651 .13021 L
.95635 .13722 L
.97619 .14108 L
s
0 0 0 r
.02381 .57368 m
.04936 .54539 L
.06027 .52869 L
.0794 .49069 L
.10317 .42161 L
.12285 .3381 L
.14473 .20469 L
.16309 .04716 L
.18254 .01544 L
.20222 .01638 L
.22158 .01472 L
.24245 .02126 L
.2619 .0297 L
.28158 .0326 L
.30221 .03898 L
.32182 .03758 L
.34127 .04996 L
.36095 .04604 L
.38094 .0694 L
.40065 .04065 L
.42063 .06475 L
.44031 .05831 L
.46031 .07947 L
.48028 .08711 L
.5 .08394 L
.51986 .06495 L
.53967 .09455 L
.55951 .08223 L
.57937 .09303 L
.59923 .08236 L
.61904 .11172 L
.63888 .09676 L
.65873 .10484 L
.67855 .12688 L
.6984 .12621 L
.71824 .11358 L
.7381 .1158 L
.75794 .13582 L
.77779 .13743 L
.79763 .14667 L
.81746 .13272 L
.8373 .1545 L
.85714 .15453 L
.87698 .15286 L
.89683 .14739 L
.91667 .15887 L
.93651 .16789 L
.95635 .16761 L
.97619 .16052 L
s
.5 .5 .5 r
.02381 .60332 m
.04936 .59227 L
.06027 .58672 L
.0794 .57234 L
.10317 .54806 L
.12285 .519 L
.14473 .47224 L
.16309 .41707 L
.18254 .33739 L
.20222 .22759 L
.22158 .07825 L
.24245 .04551 L
.2619 .10182 L
.28158 .09222 L
.30221 .08655 L
.32182 .08231 L
.34127 .12348 L
.36095 .12002 L
.38094 .14031 L
.40065 .13902 L
.42063 .14632 L
.44031 .13873 L
.46031 .177 L
.48028 .17383 L
.5 .16317 L
.51986 .15996 L
.53967 .16018 L
.55951 .17994 L
.57937 .20926 L
.59923 .16285 L
.61904 .19572 L
.63888 .20974 L
.65873 .20302 L
.67855 .23237 L
.6984 .23536 L
.71824 .23738 L
.7381 .24347 L
.75794 .2388 L
.77779 .24013 L
.79763 .27487 L
.81746 .26774 L
.8373 .27885 L
.85714 .30323 L
.87698 .2847 L
.89683 .27299 L
.91667 .29928 L
.93651 .28457 L
.95635 .2992 L
.97619 .30105 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/B000`400?l00000o`00003oo`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ
0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`00o`3ooolQ0?ooo`000`3oool0
1000100000000004000000030?ooo`030000003oool0oooo00<0oooo00D000000?ooo`3oool0oooo
000000030?ooo`040000000010000000000400<0oooo00<000@00000003oool0oP3oool000D0oooo
00<000@00?ooo`3oool02@3oool00`001000oooo000400060?ooo`050004003oool0oooo0?ooo`00
10000P3oool00`001000oooo0?ooo`0B0?ooo`030004003oool0oooo0>L0oooo00050?ooo`030004
003oool0oooo00X0oooo00<000000?ooo`3oool01@3oool01@000000oooo0?ooo`3oool000000080
oooo00<000@00?ooo`3oool04P3oool00`000000oooo0?ooo`3W0?ooo`001@3oool00`001000oooo
0?ooo`090?ooo`030004003oool000@000H0oooo00D000@00?ooo`3oool0oooo000400020?ooo`03
0004003oool0oooo0180oooo0`00103W0?ooo`001@3oool00`001000oooo0?ooo`080?ooo`050000
003oool0oooo0?ooo`0000001@3oool01@000000oooo0?ooo`3oool000000080oooo00<000@00?oo
o`3oool0103oool00`00000000@0000000020?ooo`030004003oool0oooo00H0oooo00<000000?oo
o`3oool09`3oool00`00000000@00?ooo`020?ooo`06000400000000oooo0?ooo`0010000000]@3o
ool000<0oooo0`00100B0?ooo`<000@00`3oool010001000oooo0?ooo`0010020?ooo`<000@00P3o
ool00`001000oooo00040002000400P0oooo00<000@00?ooo`3oool09`3oool<00040080oooo0P00
102`0?ooo`001@3oool00`001000oooo0?ooo`0B0?ooo`030000003oool0oooo0080oooo00<000@0
0000003oool01`3oool01@001000oooo0?ooo`00000000@000P0oooo00<000000?ooo`3oool09`3o
ool00`000000oooo0?ooo`090?ooo`04000000001000000000040;00oooo000Y0?ooo`050004003o
ool0oooo0?ooo`001000203oool00`001000oooo0?ooo`0W0?ooo`030004003oool0oooo00d0oooo
0P0010000`3oool000@00004002[0?ooo`00=P3oool00`000000oooo0?ooo`0V0?ooo`030004003o
ool0oooo0100oooo00L0000000040000000000@00?ooo`00100000000:L0oooo000f0?ooo`030004
003oool0oooo02D0oooo00<000@00?ooo`3oool05@3oool400040080oooo0P0010030?ooo`8000@0
W@3oool003H0oooo00<000000?ooo`3oool0903oool00`001000oooo0?ooo`0J0?ooo`0900000000
1000000000040000000000@000000000100000000100oooo00<000@00000003oool0RP3oool003H0
oooo00<000@00?ooo`3oool08`3oool200040100oooo0P1kOG/B0?ooo`030004003oool0oooo00D0
oooo0P0010070?ooo`8000@0R`3oool003H0oooo00<000000?ooo`3oool08`3oool00`00000000@0
0?ooo`0?0?ooo`80Ngek4`3oool02@0010000000000400000000oooo000000001000000000040006
0?ooo`030000003oool0000008/0oooo000f0?ooo`030004003oool0oooo02<0oooo00<000@00?oo
o`3oool03P3oool207]mNaH0oooo0`0010030?ooo`030004003oool0oooo00@0oooo00@000@00?oo
o`3oool000@0RP3oool003H0oooo00<000000?ooo`3oool08`3oool00`000000oooo0?ooo`0>0?oo
o`0307]mN`3oool0Ngek01`0oooo00<000000?ooo`3oool00P3oool00`001000oooo0?ooo`020?oo
o`030000003oool0oooo00<0oooo00<000000004003oool0P@3oool003H0oooo00<000@00?ooo`3o
ool08`3oool00`001000oooo0?ooo`0=0?ooo`0407]mN`3oool0oooo07]mNad0oooo00@000@00?oo
o`3oool000@01P3oool700040140oooo0P00101_0?ooo`00=P3oool00`000000oooo0?ooo`0S0?oo
o`030000003oool0oooo00d0oooo00D0Ngek0?ooo`3oool0oooo07]mN`0M0?ooo`03000000001000
000000L0oooo00<000000004003oool00P3oool00`001000oooo0?ooo`0?0?ooo`03000400000000
oooo06h0oooo000f0?ooo`030004003oool0oooo0280oooo00<000@00?ooo`3oool03@3oool00`1k
OG/0oooo0?ooo`020?ooo`0307]mN`3oool0oooo01/0oooo0P00100>0?ooo`030004003oool0oooo
00d0oooo00<000@00?l4o`0010000P3oool20?l4o`<0oooo0P3o1?mV0?ooo`00=P3oool00`000000
oooo0?ooo`0R0?ooo`030004003oool0oooo00d0oooo00<0Ngek0?ooo`3oool00P3oool00`1kOG/0
oooo0?ooo`0/0?ooo`030004003oool0oooo00P0oooo0100o`3o0?l4o`3oool000@00?l0o`3o1?l0
00000?l4o`3o0?l0o`Co0?l0o`3o1?l0o`3o0?l4o`3o0?l0o`CoIP3oool003H0oooo00<000@00?oo
o`3oool08P3oool00`001000oooo0?ooo`0;0?ooo`80Ngek1P3oool00`1kOG/0oooo0?ooo`0=0?oo
o`80Ngek7@3oool3000400L0oooo0P3o1?l00`001000oooo0?ooo`020?ooo`030004003oool0oooo
00@0oooo0P0010020?l4o`030?ooo`3o1?l0o`Co0080oooo0P00101M0?ooo`00=P3oool00`000000
oooo0?ooo`0R0?ooo`030004003oool0oooo00/0oooo0P1kOG/60?ooo`0307]mN`3oool0oooo00P0
oooo0P1kOG/00`3oool0Ngek07]mN`0207]mNah0oooo00<000@00000000010001@3oool00`3o1?l0
0000000400060?ooo`030000003oool0oooo00<0oooo00L000000004003oool0oooo0?l0o`3o1?l0
o`3o0080oooo00L000@00000003oool0o`3o0?l4o`3o0?l0o`Co0080oooo00<0003o0004o`3oool0
0P3oool00`001?l0003o0?ooo`020?ooo`030000o`001?l0oooo04T0oooo000f0?ooo`030004003o
ool0oooo0280oooo00<000@00?ooo`3oool0303oool00`1kOG/0oooo0?ooo`040?ooo`0307]mN`3o
ool0oooo00L0oooo101kOG/20?ooo`0307]mN`3oool0oooo0200oooo200010070?ooo`030004003o
ool0oooo008000@000<0oooo000400001000103oool2000400030?l4o`001000o`Co0080oooo103o
1?l<0004o`80oooo0P001?l30?ooo`8000Co@@3oool003H0oooo00<000000004003oool08P3oool0
0`001000oooo0?ooo`0<0?ooo`0307]mN`3oool0oooo00D0oooo00<0Ngek0?ooo`3oool00P3oool4
07]mN`L0oooo00<0Ngek0?ooo`3oool08@3oool00`000000o`Co0?ooo`0:0?ooo`03000400000000
00@000D0oooo00@000@000000000100000000`3oool08@000000oooo0?ooo`3oool0003o0004o`00
0?l000Co0?l0o`3o1?l0o`3o0?l4o`3o0?l0o`Co0?l0o`3o1?l0o`3o0?l4o`3o0?l0o`Co0000o`00
1?l0003o0004o`000?l000Co0000o`001?l0003o0004o`3oool000Co0000o`0m0?ooo`00=P3oool0
0`001000oooo0?ooo`0R0?ooo`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3oool01P3oool0
0`1kOG/0oooo0?ooo`020?ooo`80Ngek2@3oool00`1kOG/0oooo0?ooo`0Q0?ooo`80o`Co2`3oool2
000400P0oooo0P0010040?ooo`030004003oool0oooo008000Co0P3oool2000400<0oooo0P3o1?l5
0?ooo`D0o`Co1@3oool40004o`80oooo0P001?lh0?ooo`00=P3oool00`000000oooo0?ooo`0R0?oo
o`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3oool01`3oool407]mN``0oooo00<0Ngek0?oo
o`3oool07`3oool00`3o1?l0oooo0?ooo`0L0?ooo`03000000001?l0oooo00<0oooo00<000000004
003oool02`3oool04@3o0?l0o`Co0?ooo`3o1?l0o`3o0?l4o`3o0?l0o`Co0?ooo`3o1?l0o`3o0?oo
o`000?l000Co0000o`001?l0003o0080oooo00<000Co0000o`3oool0<P3oool003H0oooo0P00100S
0?ooo`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3oool01`3oool207]mN`h0oooo00<0Ngek
0?ooo`3oool07P3oool00`3o1?l0oooo0?ooo`0J0?ooo`<000Co00<0oooo0004003oool00P3oool2
000400040?ooo`00100000@00?ooo`8000@0303oool20?l4o`030?ooo`3o1?l0o`Co0080o`Co0P3o
ool20?l4o`030?ooo`001?l000Co00L000Co0P3oool20004oa00oooo0P00o`030?ooo`800?l04`3o
ool003H0oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`0<0?ooo`0307]mN`3oool0
oooo01L0oooo00<0Ngek0?ooo`3oool07P3oool00`3o0?l0oooo0?ooo`0J0?ooo`080004o`000?l0
oooo0?ooo`00100000000004000000050?ooo`03000000001000000001<0oooo00@0o`3o0?l4o`3o
0?l0o`Co1`3oool020001?l0003o0?ooo`000?l000Co0000o`001?l0003o0P3oool00`001?l0003o
0?ooo`060?ooo`800?l000<0oooo003o0000o`001`00o`020?ooo`800?l03P3oool003H0oooo0P00
100R0?ooo`030004003oool0oooo00`0oooo00<0Ngek0?ooo`3oool0603oool00`1kOG/0oooo0?oo
o`0L0?ooo`030?l4o`3oool0oooo01X0oooo00<000Co0?ooo`3oool00`3oool2000400P0oooo00<0
00@00?ooo`3oool02P3oool200040080oooo0P0010050?ooo`80o`Co00<0oooo0?l4o`3o1?l01`3o
ool20?l4o`030?ooo`001?l000Co00L000Co1@00o`090?ooo`@00?l00P3oool2003o00<0oooo0P00
o`050?ooo`000`3oool01000100000000004000000030?ooo`030000003oool0oooo00<0oooo00D0
00000?ooo`3oool0oooo000000030?ooo`040000000010000000000400<0oooo00<000@00000003o
ool0503oool00`00000000@00?ooo`0Q0?ooo`030000003oool0oooo00`0oooo00<0Ngek0?ooo`3o
ool0603oool00`1kOG/0oooo0?ooo`030?ooo`80Ngek5@3oool00`3o1?l0o`3o0?ooo`0K0?ooo`03
0004o`3oool0oooo00l0oooo00<000@00?ooo`3oool0203oool01`00000000@00000000010000000
0004000000001`3oool00`3o0?l0o`Co0?l0o`020?ooo`080?l4o`3o0?l0oooo0?l0o`3o1?l0o`3o
0?l4o`3o0?l20?ooo`030?l4o`3o0?l0oooo00@00?l000P0003o0004o`000?l000Co0000o`3oool0
003o0004o`X0oooo2@00o`050?ooo`001@3oool00`001000oooo0?ooo`090?ooo`030004003oool0
00@000H0oooo00D000@00?ooo`3oool0oooo000400020?ooo`030004003oool0oooo0180oooo0P00
100R0?ooo`030004003oool0oooo00`0oooo00<0Ngek0?ooo`3oool06@3oool207]mN`030?ooo`1k
OG/0Ngek0080Ngek5@3oool20?l4oaX0oooo0P001?lC0?ooo`030004003oool0oooo00D0oooo0P00
10070?ooo`030004003oool0oooo00L0oooo1@3o1?l50?ooo`T0o`Co0P3oool20?l4o`030?ooo`00
1?l000Co008000Co0P3oool20004o`<0oooo0P001?l>0?ooo`001@3oool00`001000oooo0?ooo`0:
0?ooo`030000003oool0oooo00D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030004003o
ool0oooo0180oooo00<000000004003oool08@3oool00`000000oooo0?ooo`0;0?ooo`0307]mN`3o
ool0oooo01X0oooo0`1kOG/30?ooo`0307]mN`3oool0oooo0180oooo00<0o`3o0?ooo`3oool06P3o
ool00`001?l0003o0?ooo`0C0?ooo`0700040000000000@000000000100000000004000:0?ooo`03
0000003oool0oooo0140oooo0`00o`050?ooo`0B0?l0o`3o1?l0o`3o0?l4o`3o0?l0oooo0?l0o`3o
1?l0oooo0004o`000?l000Co0000o`001?l0003o0004o`000?l000Co3P3oool000D0oooo00<000@0
0?ooo`3oool02@3oool00`001000oooo000400060?ooo`050004003oool0oooo0?ooo`0010000P3o
ool00`001000oooo0?ooo`0B0?ooo`<000@08@3oool00`001000oooo0?ooo`0;0?ooo`0307]mN`3o
ool0oooo0240oooo00<0Ngek0?ooo`3oool0403oool00`3o1?l0oooo0?ooo`0J0?ooo`030004o`3o
ool0oooo01D0oooo0P0010020?ooo`8000@02P3oool00`001000oooo0?ooo`0=0?ooo`8000@00P00
o`0=0?ooo`@0o`Co0P3oool20?l4o`030?ooo`3o1?l0o`Co0080o`Co1@001?l20?ooo`8000Co1@3o
ool000D0oooo00<000@00?ooo`3oool0203oool01@000000oooo0?ooo`3oool0000000D0oooo00D0
00000?ooo`3oool0oooo000000020?ooo`030004003oool0oooo0180oooo00<000000?ooo`3oool0
8@3oool00`000000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo0280oooo00<0Ngek0?ooo`3oool0
3P3oool00`3o0?l0oooo0?ooo`0I0?ooo`030000o`001?l0oooo02P0oooo00<000@0000000001000
0`3oool00`00100000000?ooo`060?ooo`0300000000o`00oooo0180oooo00D0o`Co0?l0o`3o1?l0
o`3o0?l4o`020?ooo`0;0?l0o`3o1?l0o`3o0?ooo`3o0?l0o`Co0?ooo`001?l0003o0004o`000?l0
1@3oool000<0oooo0`00100B0?ooo`<000@00`3oool010001000oooo0?ooo`0010060?ooo`<000@0
00@0oooo00040000100000@01`3oool00`001000oooo0?ooo`0Q0?ooo`030004003oool0oooo00/0
oooo00<0Ngek0?ooo`3oool08`3oool207]mN`030?ooo`1kOG/0Ngek0080Ngek1P3oool207]mN`03
0?l4o`3oool0oooo01P0oooo0P001?l/0?ooo`L000@01@3oool2000400030?ooo`001000oooo01`0
oooo103o1?l20?ooo`80o`Co1@3oool000D0oooo00<000@00?ooo`3oool04P3oool00`000000oooo
0?ooo`020?ooo`03000400000000oooo0080oooo00<000000004000000000P3oool01@001000oooo
0?ooo`3oool000@000P0oooo00<000000?ooo`3oool08@3oool00`000000oooo0?ooo`0;0?ooo`03
07]mN`3oool0oooo02<0oooo0`1kOG/20?ooo`@0Ngek00L0oooo07]mN`1kOG/0Ngek0?ooo`3o0?l0
Ngek01X0oooo00<0003o0004o`3oool0<@3oool01P00000000@00?ooo`0010000000000400@0oooo
00<000000?ooo`3oool07P3oool0103o1?l0o`3o0?l4o`3o0?l50?ooo`00:@3oool01@001000oooo
0?ooo`3oool000@000P0oooo00<000@00?ooo`3oool08@3oool00`001000oooo0?ooo`0;0?ooo`03
07]mN`3oool0oooo02`0oooo0`1kOG/0103oool0o`Co0?ooo`1kOG/I0?ooo`030004o`3oool0oooo
03@0oooo0`0010060?ooo`030004003oool0oooo00`0oooo0P00100H0?ooo`00:@3oool01@001000
oooo0?ooo`3oool000@000P0oooo00<000000?ooo`3oool0803oool00`001000oooo0?ooo`0<0?oo
o`0307]mN`3oool0oooo0300oooo00@0o`Co0?ooo`3oool0Ngek5`3oool00`000?l0oooo0?ooo`0c
0?ooo`800?l02P3oool00`00000000@0000000020?ooo`03000400000000oooo0080oooo00L00000
00040000000000@0000000001000000001L0oooo000f0?ooo`030004003oool0oooo0200oooo00<0
00@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0`0?ooo`050?l4o`3oool0oooo0?ooo`1kOG/0
503oool30004ocD0oooo0P00o`0<0?ooo`/000@01@3oool4000401<0oooo000f0?ooo`030000003o
ool0oooo0200oooo00<000@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0_0?ooo`030?l4o`3o
ool0oooo00<0oooo00<0Ngek0?ooo`3oool0203oool207]mN`030?ooo`1kOG/0Ngek00H0Ngek303o
ool207]mNbH0oooo1000o`0N0?ooo`03000400000000oooo00X0oooo00<000000004000000001@3o
ool003H0oooo00<000@00?ooo`3oool0803oool00`001000oooo0?ooo`0;0?ooo`0307]mN`3oool0
oooo02h0oooo0P3o1?l50?ooo`0307]mN`3oool0oooo00P0oooo0`1kOG/20?ooo`80Ngek0P3oool0
0`001?l0Ngek07]mN`0;0?ooo`80Ngek8`3oool3003o02@0oooo0P0010000`3oool000@000040002
0?ooo`@000@000<0oooo0004000010001@3oool003H0oooo00<000000?ooo`3oool0803oool00`00
1000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo02h0oooo00<0o`3o0?l4o`3oool01@3oool00`1k
OG/0oooo0?ooo`050?ooo`80Ngek1`3oool00`001?l0003o0?ooo`020?ooo`0307]mN`3oool0oooo
00P0oooo0P1kOG/S0?ooo`800?l09`3oool01`001000000000040000000000@00000000010002P3o
ool003H0oooo00<000@00?ooo`3oool0803oool00`001000oooo0?ooo`0;0?ooo`0307]mN`3oool0
oooo02d0oooo00<0o`Co0?ooo`3oool01P3oool307]mN`<0oooo0P1kOG/70?ooo`8000Co1@3oool0
0`1kOG/0oooo0?ooo`070?ooo`0307]mN`3oool0Ngek0240oooo0P00o`0j0?ooo`00=P3oool00`00
0000oooo0?ooo`0P0?ooo`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3oool0;03oool00`3o
0?l0oooo0?ooo`080?ooo`L0Ngek1`3oool00`001?l0003o0?ooo`050?ooo`<0Ngek1P3oool0101k
OG/0oooo0?ooo`1kOG/M0?ooo`<00?l0?03oool003H0oooo00<000@00?ooo`3oool0803oool00`00
1000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo02`0oooo00<0o`Co0?ooo`3oool05@3oool00`00
1?l0oooo0?ooo`070?ooo`80Ngek1P3oool0101kOG/0oooo0?ooo`1kOG/M0?ooo`800?l0?@3oool0
03H0oooo00<000000?ooo`3oool0803oool00`001000oooo0?ooo`0:0?ooo`0307]mN`3oool0oooo
02`0oooo00<0o`Co0?ooo`3oool0503oool00`000?l000Co0?ooo`0:0?ooo`0307]mN`3oool0oooo
00<0oooo00<0Ngek0?ooo`3oool00P3oool00`1kOG/0oooo0?ooo`0I0?ooo`800?l0?P3oool003H0
oooo00<000@00?ooo`3oool07`3oool00`001000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo02X0
oooo0P3o1?lF0?ooo`030004o`3oool0oooo00/0oooo00<0Ngek0?ooo`3oool00`3oool00`1kOG/0
oooo0?ooo`030?ooo`0307]mN`3oool0oooo01D0oooo0`00o`100?ooo`00=P3oool00`000000oooo
0?ooo`0O0?ooo`030000003oool0oooo00/0oooo00<0Ngek0?ooo`3oool0:P3oool00`3o1?l0o`3o
0?ooo`0D0?ooo`030000o`001?l0oooo00d0oooo00D0Ngek0?ooo`3oool0oooo07]mN`060?ooo`<0
Ngek5@3oool2003o0440oooo000f0?ooo`030004003oool0oooo01l0oooo00<000@00?ooo`3oool0
2`3oool00`1kOG/0oooo0?ooo`0Y0?ooo`030?l4o`3oool0oooo01@0oooo0`001?l?0?ooo`0407]m
N`3oool0oooo07]mN`L0oooo0P1kOG/C0?ooo`800?l0@`3oool003H0oooo00<000000?ooo`3oool0
7`3oool00`00000000@00?ooo`0;0?ooo`0307]mN`3oool0oooo02P0oooo00<0o`Co0?ooo`3oool0
503oool00`000?l0oooo0?ooo`0A0?ooo`0307]mN`3oool0Ngek00P0oooo0P1kOG/50?ooo`<0Ngek
1`3oool3003o04D0oooo000f0?ooo`030004003oool0oooo01l0oooo0P00100<0?ooo`0307]mN`3o
ool0oooo02P0oooo00<0o`Co0?ooo`3oool0503oool00`001?l0oooo0?ooo`0A0?ooo`<0Ngek2P3o
ool507]mN`030?ooo`1kOG/0Ngek00L0oooo0P00o`160?ooo`00=P3oool00`00000000@00?ooo`0O
0?ooo`030000003oool0oooo00/0oooo00<0Ngek0?ooo`3oool09@3oool00`3o0?l0o`Co0?l0o`0E
0?ooo`030000o`001?l0oooo01@0oooo0P1kOG/;0?ooo`80Ngek103oool00`1kOG/0oooo0?ooo`03
0?ooo`800?l0B03oool003H0oooo00<000@00?ooo`3oool07`3oool00`001000oooo0?ooo`0:0?oo
o`0307]mN`3oool0oooo02H0oooo0P3o1?lF0?ooo`8000Co:@3oool00`1kOG/0oooo0?ooo`02003o
04X0oooo000f0?ooo`030000003oool0oooo01l0oooo00<000000?ooo`3oool02P3oool00`1kOG/0
oooo0?ooo`0V0?ooo`030?l0o`3oool0oooo01@0oooo00<000Co0?ooo`3oool0:P3oool0101kOG/0
0?l0003o0000o`1:0?ooo`00=P3oool2000401l0oooo00<000@00?ooo`3oool02`3oool207]mNbH0
oooo00<0o`Co0?ooo`3oool04`3oool20004ob`0oooo0P00o`000`1kOG/0oooo0?ooo`1:0?ooo`00
=P3oool00`000000oooo0?ooo`0N0?ooo`030004003oool0oooo00/0oooo0P1kOG/V0?ooo`030?l4
o`3oool0oooo0140oooo00<000Co0000o`3oool0:`3oool2003o0080oooo0`1kOG]:0?ooo`00=P3o
ool2000401l0oooo00<000@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0T0?ooo`030?l4o`3o
ool0oooo0180oooo0P001?lZ0?ooo`@00?l00`3oool207]mN`80oooo0P1kOG]60?ooo`00=P3oool0
0`00000000@00?ooo`0N0?ooo`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3oool08P3oool0
0`3o0?l0o`Co0?ooo`0C0?ooo`030000o`3oool0oooo02L0oooo0`00o`090?ooo`T0Ngek1`3oool2
07]mN`<0oooo0P1kOG/c0?ooo`001`3oool4000400<0oooo00<000@00?ooo`3oool0103oool01@00
1000oooo0?ooo`3oool000@00080oooo100010030?ooo`8000@04@3oool2000401l0oooo00<000@0
0?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0R0?ooo`80o`Co4P3oool20004obX0oooo0P00o`0A
0?ooo`H0Ngek00<0oooo07]mN`1kOG/01`1kOG/c0?ooo`002@3oool00`001000oooo0?ooo`0:0?oo
o`030000003oool0000000D0oooo00D000000?ooo`3oool0oooo000000020?ooo`030004003oool0
oooo00h0oooo00<000000004003oool07P3oool00`001000oooo0?ooo`0:0?ooo`0307]mN`3oool0
oooo0280oooo00<0o`Co0?ooo`3oool04@3oool00`001?l0oooo0?ooo`0Y0?ooo`800?l05P3oool3
07]mN`L0oooo00<0Ngek0?ooo`3oool0<P3oool000T0oooo00<000@00?ooo`3oool02`3oool00`00
1000oooo0?ooo`040?ooo`050004003oool0oooo0?ooo`0010000P3oool00`001000oooo0?ooo`0>
0?ooo`<000@07@3oool00`001000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo0240oooo00<0o`Co
0?ooo`3oool04@3oool20004obP0oooo0`00o`0S0?ooo`0307]mN`3oool0oooo0340oooo00090?oo
o`030004003oool0oooo00X0oooo00<000000?ooo`0000001@3oool01@000000oooo0?ooo`3oool0
00000080oooo00<000@00?ooo`3oool03P3oool00`000000oooo0?ooo`0M0?ooo`030000003oool0
oooo00/0oooo00<0Ngek0?ooo`3oool08@3oool00`3o0?l0oooo0?ooo`0@0?ooo`030004o`000?l0
00Co02P0oooo0P00o`0T0?ooo`0307]mN`3oool0oooo0340oooo00090?ooo`030004003oool0oooo
00T0oooo00D000@00?ooo`3oool0oooo000400040?ooo`050004003oool0oooo0?ooo`0010000P3o
ool00`001000oooo0?ooo`050?ooo`<000@01P3oool00`001000oooo0?ooo`0M0?ooo`030004003o
ool0oooo00/0oooo00<0Ngek0?ooo`3oool0803oool00`3o1?l0oooo0?ooo`0@0?ooo`030004o`3o
ool0oooo02L0oooo0P00o`0W0?ooo`0307]mN`3oool0oooo0300oooo00070?ooo`03000400000000
00@00180oooo00<000000004000000000`3oool010000000oooo0?ooo`0010020?ooo`0500000000
100000000?ooo`0000000P3oool00`001000oooo0?ooo`040?ooo`030000003oool0oooo01d0oooo
00<000000?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0N0?ooo`030?l4o`3o0?l0oooo0180oooo
00<0003o0?ooo`3oool0903oool3003o02T0oooo00<0Ngek0?ooo`3oool0<03oool000T0oooo00<0
00@00?ooo`3oool04P3oool00`001000oooo0?ooo`020?ooo`8000@0203oool2000400L0oooo00<0
00@00?ooo`3oool07@3oool00`001000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo01h0oooo0P3o
1?lA0?ooo`8000Co9`3oool2003o02/0oooo00<0Ngek0?ooo`3oool00P3oool307]mNbX0oooo000]
0?ooo`0300040000000000@000H0oooo00<000000?ooo`3oool07@3oool00`000000oooo0?ooo`0:
0?ooo`0307]mN`3oool0oooo01h0oooo00<0o`3o0?ooo`3oool04@3oool00`000?l000Co0?ooo`0T
0?ooo`800?l0;@3oool507]mN`040?ooo`1kOG/0Ngek07]mN`l0oooo0P1kOG/H0?ooo`00=P3oool0
0`001000oooo0?ooo`0L0?ooo`030004003oool0oooo00/0oooo00<0Ngek0004000010000P001003
0?ooo`8000@0103oool200040100oooo00<0o`Co0?ooo`3oool04@3oool00`001?l0oooo0?ooo`05
0?ooo`@000@00`3oool2000400@0oooo0P0010040?ooo`8000@02@3oool2003o01X0oooo10001003
0?ooo`8000@0103oool2000400@0oooo0P001000103oool0Ngek07]mN`3oool2000400<0oooo101k
OG/;0?ooo`80Ngek603oool003H0oooo00<000000?ooo`3oool0703oool00`001000oooo0?ooo`0;
0?ooo`0407]mN`3oool0oooo000000<0oooo00@000000?ooo`3oool000@00P3oool010000000oooo
0?ooo`00100?0?ooo`030?l4o`3oool0oooo00l0oooo00<000Co0000o`3oool02@3oool01@001000
oooo0?ooo`3oool000@00080oooo00@000000?ooo`3oool000@00P3oool010000000oooo0?ooo`00
10020?ooo`030000003oool0oooo00H0oooo0P00o`0L0?ooo`050004003oool0oooo0?ooo`001000
0P3oool010000000oooo0?ooo`0010020?ooo`040000003oool0oooo00040080oooo00@000000?oo
o`3oool000@00P3oool00`000000oooo0?ooo`020?ooo`80Ngek2@3oool207]mN`80oooo00<0Ngek
0?ooo`3oool01@3oool207]mN`h0oooo000f0?ooo`030004003oool0oooo01`0oooo00<000@00?oo
o`3oool02`3oool0101kOG/0oooo0?ooo`0010030?ooo`040004003oool0oooo00040080oooo00@0
00@00?ooo`3oool000@03@3oool20?l4oa00oooo0P001?l<0?ooo`050004003oool0oooo0?ooo`00
10000P3oool010001000oooo0?ooo`0010020?ooo`040004003oool0oooo00040080oooo00<000@0
0?ooo`3oool0103oool2003o01h0oooo00D000@00?ooo`3oool0oooo000400020?ooo`040004003o
ool0oooo00040080oooo00@000@00?ooo`3oool000@00P3oool010001000oooo0?ooo`0010020?oo
o`030004003oool0oooo00<0oooo00<0Ngek0?ooo`3oool01@3oool207]mN`@0oooo00<0Ngek0?oo
o`3oool01@3oool207]mN`h0oooo000f0?ooo`030000003oool0oooo01`0oooo00<000@00?ooo`3o
ool02`3oool0101kOG/0oooo0?ooo`0000030?ooo`040000003oool0oooo00040080oooo00@00000
0?ooo`3oool000@03@3oool00`3o1?l0o`3o0?ooo`0?0?ooo`030004o`000?l0oooo00/0oooo00D0
00@00?ooo`3oool0oooo000400020?ooo`040000003oool0oooo00040080oooo00@000000?ooo`3o
ool000@00P3oool010000000oooo0?ooo`3oool3003o0200oooo00D000@00?ooo`3oool0oooo0004
00020?ooo`040000003oool0oooo00040080oooo00@000000?ooo`3oool000@00P3oool010000000
oooo0?ooo`0010020?ooo`030000003oool0oooo00<0oooo00<0Ngek0?ooo`3oool0103oool307]m
N`D0oooo00<0Ngek0?ooo`3oool00`3oool00`1kOG/0oooo07]mN`0>0?ooo`00=P3oool00`001000
oooo0?ooo`0L0?ooo`030004003oool0oooo00/0oooo00@0Ngek0?ooo`3oool000@00`3oool01000
1000oooo0?ooo`0010020?ooo`040004003oool0oooo000400`0oooo00<0o`Co0?ooo`3oool03`3o
ool00`001?l0oooo0?ooo`0<0?ooo`050004003oool0oooo0?ooo`0010000P3oool010001000oooo
0?ooo`0010020?ooo`040004003oool0oooo00040080oooo00@000@00?ooo`3oool0oooo0P00o`0Q
0?ooo`050004003oool0oooo0?ooo`0010000P3oool010001000oooo0?ooo`0010020?ooo`040004
003oool0oooo00040080oooo00@000@00?ooo`3oool000@00P3oool00`001000oooo0?ooo`040?oo
o`0307]mN`3oool0oooo0080oooo00<0Ngek0?ooo`3oool01`3oool00`1kOG/0oooo0?ooo`0207]m
N`<0oooo00<0Ngek0?ooo`3oool02`3oool003H0oooo00<000000?ooo`3oool0703oool00`001000
oooo0?ooo`0;0?ooo`0407]mN`00000000@0000000<0oooo00@000000?ooo`3oool000@00P3oool0
10000000oooo0?ooo`00100;0?ooo`030?l4o`3oool0oooo00l0oooo00<000Co0?ooo`3oool02`3o
ool00`0010000000000400030?ooo`040004003oool0oooo00000080oooo00@000@00?ooo`3oool0
00000P3oool01@001000oooo0?ooo`000000oooo00800?l08@3oool00`0010000000000400030?oo
o`040004003oool0oooo00000080oooo00@000@00?ooo`3oool000000P3oool010001000oooo0?oo
o`0000020?ooo`040004003oool0oooo000000L0oooo00@0Ngek0?ooo`3oool0Ngek2P3oool307]m
N`H0oooo0`1kOG/30?ooo`80Ngek1@3oool003H0oooo00<000@00?ooo`3oool06`3oool00`001000
oooo0?ooo`0;0?ooo`0507]mN`3oool0oooo0?ooo`001000103oool2000400@0oooo0P00100<0?oo
o`030?l4o`3oool0oooo00`0oooo0`001?l@0?ooo`030004003oool0oooo0080oooo0P0010040?oo
o`8000@0103oool0100010000?l0003o0000o`0U0?ooo`030004003oool0oooo0080oooo0P001004
0?ooo`8000@0103oool2000400@0oooo0P0010080?ooo`<0Ngek303oool207]mN`L0oooo1`1kOG/5
0?ooo`00=P3oool00`000000oooo0?ooo`0K0?ooo`030000003oool0oooo00/0oooo00<0Ngek0?oo
o`3oool06@3oool00`3o0?l0oooo0?ooo`0=0?ooo`030000o`001?l0oooo0280oooo0P00o`180?oo
o`80Ngek8@3oool003H0oooo00<000@00?ooo`3oool06`3oool00`001000oooo0?ooo`0;0?ooo`03
07]mN`3oool0oooo01L0oooo0P3o1?l@0?ooo`030004o`3oool0oooo0200oooo0P00o`1]0?ooo`00
=P3oool0N@00000000@0000000001000000000040000000000@00000000010000000000400000000
00@0000000001000000000040000000000@0000000001000000000040000000000@0000000001000
000000040000000000@0000000001000000000040000000000@00000000010000000000400000000
00@007]mN`001000000000040000000000@0000000001000000000040000000000@0000000001000
000000040000000000@0000000001000000000040000000000@0000000001000o`3o0?l4o`000000
00@0000000001000000000040000000000@0000000001000000000040000000000@00000o`001?l0
000000040000000000@0000000001000000000040000000000@00000000010000000000400000000
00@0000000001000000000040000000000@0000000001000000000040000000000@0000000001000
00000004000000000P00o`00KP001000000000040000000000@00000000010000000000400000000
00@0000000001000000000040000000000@0000000001000000000040000000000@0000000001000
000000040000000000@0000000001000000000040000000000@00000000010000000000400000000
00@0000000001000000000040000000000@0000000001000000000040000000000@0000000001000
000000040000000000@0000000001000000000040000000000@00000000010000000000400000000
00@0000000001000000000040000000000@0000000001000000000040000000000@0000000001000
000000040000000000@0000000001000000000040000000000@00000000010000000000400000000
00@0000000001000000000040000000000@000000000100000000@00100003H0oooo00<000@00?oo
o`3oool02@3oool00`001000oooo0?ooo`070?ooo`030004003oool0oooo00D0oooo00<000@00?oo
o`3oool00`3oool00`001000oooo0?ooo`020?ooo`050004003oool0oooo07]mN`0010000P3oool0
1`001000oooo0?ooo`3oool000@00?ooo`0010003P3oool01@3o1?l0oooo0?ooo`3oool000@000X0
oooo00<000@00004o`3oool0103oool00`001000oooo0?ooo`030?ooo`030004003oool0oooo0080
oooo00D000@00?ooo`3oool0oooo000400030?ooo`040004003oool0oooo00040080oooo00<000@0
0?ooo`3oool01000o`0;0?ooo`030004003oool0oooo00P0oooo00<000@00?ooo`3oool01@3oool0
0`001000oooo0?ooo`020?ooo`030004003oool0oooo0080oooo00D000@00?ooo`3oool0oooo0004
00030?ooo`040004003oool0oooo00040080oooo00<000@00?ooo`3oool03`3oool00`001000oooo
0?ooo`080?ooo`030004003oool0oooo00D0oooo00<000@00?ooo`3oool00`3oool00`001000oooo
0?ooo`020?ooo`050004003oool0o`@00?l4000010000P3oool100040040oooo000f0?ooo`030000
003oool0oooo01/0oooo00<000000?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0E0?ooo`030?l0
o`3oool0oooo00d0oooo00<0003o0004o`3oool07@3oool3003o06L0oooo00L0o`000?l4003oool0
o`@00?l0003o1000o`0000D0oooo000f0?ooo`030004003oool0oooo01X0oooo00<000@00?ooo`3o
ool02`3oool00`1kOG/0oooo0?ooo`0F0?ooo`030?l4o`3oool0oooo00`0oooo0`001?lN0?ooo`80
0?l0IP3oool50?l400T0oooo000f0?ooo`030000003oool0oooo01X0oooo00<000@00?ooo`3oool0
2`3oool00`1kOG/0oooo0?ooo`0E0?ooo`030?l4o`3oool0oooo00`0oooo00<0003o0?ooo`3oool0
7@3oool2003o06D0oooo00<0o`@00?l0003o10003P3oool003H0oooo0P00100J0?ooo`8000@0303o
ool207]mNaD0oooo0P3o1?l>0?ooo`030004o`3oool0oooo01`0oooo0P00o`1S0?ooo`@0o`@000<0
oooo0?l4003o10003P3oool003H0oooo00<000000?ooo`3oool06@3oool00`00000000@00?ooo`0;
0?ooo`80Ngek5@3oool00`3o1?l0o`3o0?ooo`0;0?ooo`030004o`000?l0oooo01h0oooo0P00o`1Q
0?ooo`040?l4003o0000o`@00?l001<0oooo000f0?ooo`030004003oool0oooo01X0oooo00<000@0
0?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0B0?ooo`030?l4o`3oool0oooo00`0oooo0P001?lM
0?ooo`800?l0H03oool30?l401L0oooo000f0?ooo`03000000001000oooo01T0oooo00<000000?oo
o`3oool02`3oool00`1kOG/0oooo0?ooo`0C0?ooo`030?l0o`3oool0oooo00/0oooo00<0003o0?oo
o`3oool06P3oool3003o05h0oooo00H0o`000?l4003o0000o`@00?l0003o100H0?ooo`00=P3oool2
000401X0oooo00<000@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0B0?ooo`030?l4o`3oool0
oooo00/0oooo00<000Co0?ooo`3oool06`3oool2003o05X0oooo0P3o10000`3oool0o`@00?l40002
0?l401`0oooo000f0?ooo`030000003oool0oooo01T0oooo00<000000?ooo`3oool02`3oool00`1k
OG/0oooo0?ooo`0@0?ooo`030?l4o`3o0?l0oooo00/0oooo00<000Co0000o`3oool06`3oool2003o
05/0oooo00@0o`000?l4003o0000o`@0803oool003H0oooo0P00100J0?ooo`030004003oool0oooo
00/0oooo00<0Ngek0?ooo`3oool0403oool20?l4o``0oooo0P001?lI0?ooo`<00?l0FP3oool30?l4
02@0oooo000f0?ooo`03000000001000oooo01P0oooo00<000@00?ooo`3oool02`3oool00`1kOG/0
oooo0?ooo`0@0?ooo`030?l0o`3oool0oooo00/0oooo00<0003o0?ooo`3oool06@3oool2003o05H0
oooo00L0o`000?l4003oool0o`@00?l0003o1000o`0002D0oooo00090?ooo`8000@0103oool00`00
1000oooo0?ooo`030?ooo`8000@0103oool2000400@0oooo0P0010040?ooo`8000@00`3oool40004
00H0oooo0P00100I0?ooo`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3oool03`3oool00`3o
1?l0oooo0?ooo`0:0?ooo`8000Co6P3oool2003o05H0oooo1@3o100Y0?ooo`00203oool010000000
oooo0?ooo`0010080?ooo`040000003oool0oooo00040080oooo00@000000?ooo`3oool000@00P3o
ool010000000oooo0?ooo`0010020?ooo`040000003oool0oooo000400@0oooo00<000000?ooo`3o
ool01@3oool00`00000000@00000000H0?ooo`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3o
ool03P3oool00`3o0?l0oooo0?ooo`090?ooo`030000o`001?l0oooo01T0oooo0P00o`1D0?ooo`04
0?l0003o1000o`000?l402h0oooo00080?ooo`040004003oool0oooo000400P0oooo00@000@00?oo
o`3oool000@00P3oool010001000oooo0?ooo`0010020?ooo`040004003oool0oooo00040080oooo
00@000@00?ooo`3oool000@0103oool00`001000oooo0?ooo`050?ooo`030004003oool0oooo01P0
oooo00<000@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0=0?ooo`030?l4o`3oool0oooo00X0
oooo0P001?lI0?ooo`<00?l0D@3oool30?l400030?ooo`3o1000o`@002l0oooo00080?ooo`040000
003oool0oooo000400P0oooo00@000000?ooo`3oool000@00P3oool010000000oooo0?ooo`001002
0?ooo`040000003oool0oooo00040080oooo00@000000?ooo`3oool000@0103oool00`000000oooo
0?ooo`050?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool02`3oool00`1kOG/0oooo
0?ooo`0=0?ooo`030?l0o`3o1?l0oooo00X0oooo00<000Co0?ooo`3oool05`3oool2003o04l0oooo
00L0o`000?l4003oool0o`@00?l0003o1000o`0003<0oooo00080?ooo`040004003oool0oooo0004
00P0oooo00@000@00?ooo`3oool000@00P3oool010001000oooo0?ooo`0010020?ooo`040004003o
ool0oooo00040080oooo00@000@00?ooo`3oool000@0103oool00`001000oooo0?ooo`050?ooo`03
0004003oool0oooo01L0oooo00<000@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0<0?ooo`<0
o`Co2P3oool00`001?l0oooo0?ooo`0F0?ooo`800?l0D03oool40?l403L0oooo00080?ooo`040000
003oool0oooo000400P0oooo00@000000?ooo`3oool000@00P3oool010000000oooo0?ooo`001002
0?ooo`040000003oool0oooo00040080oooo00@000000?ooo`3oool000@00P3oool00`00000000@0
000000070?ooo`030000003oool0oooo01L0oooo00<000000?ooo`3oool02`3oool00`1kOG/0oooo
0?ooo`0<0?ooo`030?l4o`3oool0oooo00T0oooo00<000Co0?ooo`3oool05@3oool4003o04`0oooo
00@0o`@00?l0003o1000o`00>`3oool000T0oooo0P00100:0?ooo`8000@0103oool2000400@0oooo
0P0010040?ooo`8000@01@3oool00`001000oooo0?ooo`050?ooo`030004003oool0oooo01L0oooo
00<000@00?ooo`3oool02P3oool00`1kOG/0oooo0?ooo`0<0?ooo`030?l4o`3oool0oooo00P0oooo
0P001?lE0?ooo`<00?l0C03oool20?l400040?ooo`3o1000o`@00?l403d0oooo000f0?ooo`030000
003oool0oooo01H0oooo00<000@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`0;0?ooo`030?l4
o`3oool0oooo00T0oooo00<000Co0000o`3oool0503oool2003o04`0oooo00@0o`000?l4003o0000
o`@0@03oool003H0oooo00<000@00?ooo`3oool05P3oool00`001000oooo0?ooo`0;0?ooo`0307]m
N`3oool0oooo00/0oooo00<0o`Co0?ooo`3oool0203oool00`001?l0oooo0?ooo`0D0?ooo`800?l0
B@3oool40?l404@0oooo000f0?ooo`030000003oool0oooo01H0oooo00<000@00?ooo`3oool02`3o
ool00`1kOG/0oooo0?ooo`090?ooo`030?l4o`3o0?l0oooo00T0oooo00<000Co0?ooo`3oool04P3o
ool3003o04L0oooo00H0o`000?l4003o0000o`@00?l0003o10160?ooo`00=P3oool00`001000oooo
0?ooo`0F0?ooo`030004003oool0oooo00X0oooo0P1kOG/;0?ooo`80o`Co203oool20004oaD0oooo
0P00o`160?ooo`@0o`@0BP3oool003H0oooo00<000000?ooo`3oool05@3oool00`00000000@00?oo
o`0;0?ooo`80Ngek2P3oool00`3o0?l0oooo0?ooo`080?ooo`030004o`000?l0oooo0180oooo0P00
o`140?ooo`040?l0003o1000o`000?l404h0oooo000f0?ooo`030004003oool0oooo01D0oooo0P00
100<0?ooo`0307]mN`3oool0oooo00T0oooo00<0o`Co0?ooo`3oool01`3oool00`001?l0oooo0?oo
o`0@0?ooo`<00?l0@P3oool40?l400030?ooo`3o1000o`@004l0oooo000f0?ooo`030000003oool0
oooo01D0oooo00<000000?ooo`3oool02P3oool00`1kOG/0oooo0?ooo`090?ooo`030?l4o`3oool0
oooo00H0oooo00<0003o0004o`3oool04P3oool2003o03l0oooo00H0o`000?l4003oool0o`@00?l0
003o101D0?ooo`00=P3oool00`001000oooo0?ooo`0D0?ooo`030004003oool0oooo00/0oooo00<0
Ngek0?ooo`3oool01`3oool20?l4o`L0oooo0P001?lC0?ooo`800?l0@03oool40?l405L0oooo000f
0?ooo`030000003oool0oooo01@0oooo00<000@00?ooo`3oool02`3oool00`1kOG/0oooo0?ooo`07
0?ooo`030?l4o`3o0?l0oooo00H0oooo00<0003o0004o`3oool0403oool2003o03h0oooo00@0o`@0
0?l0003o1000o`00F`3oool003H0oooo00<000@00?ooo`3oool0503oool00`001000oooo0?ooo`0:
0?ooo`0307]mN`3oool0oooo00L0oooo00<0o`Co0?ooo`3oool01P3oool00`001?l0oooo0?ooo`0A
0?ooo`800?l0>P3oool60?l405d0oooo000f0?ooo`030000003oool0oooo01<0oooo00<000000?oo
o`3oool02`3oool00`1kOG/0oooo0?ooo`060?ooo`030?l4o`3oool0oooo00H0oooo00<0003o0?oo
o`3oool0403oool2003o03/0oooo00<0o`000?l4003o0000H@3oool003H0oooo00<000@00?ooo`3o
ool04`3oool00`001000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo00D0oooo00<0o`Co0?ooo`3o
ool0103oool30004oa00oooo0`00o`0i0?ooo`@0o`@0I03oool003H0oooo00<000000?ooo`3oool0
4`3oool00`000000oooo0?ooo`0;0?ooo`0307]mN`3oool0oooo00<0oooo00<0o`3o0?l4o`3oool0
1P3oool00`001?l0003o0?ooo`0?0?ooo`<00?l0=@3oool01`3o1000o`000?ooo`3o0000o`@00?l0
003o1000IP3oool003H0oooo0P00100D0?ooo`030004003oool0oooo00X0oooo00<0Ngek0?ooo`3o
ool0103oool20?l4o`L0oooo00<000Co0?ooo`3oool03@3oool2003o03L0oooo103o101Z0?ooo`00
=P3oool00`000000oooo0?ooo`0B0?ooo`030004003oool0oooo00/0oooo00<0Ngek0?ooo`3oool0
0`3oool00`3o1?l0oooo0?ooo`050?ooo`030004o`000?l0oooo00d0oooo0P00o`0e0?ooo`040?l0
003o1000o`000?l406h0oooo000f0?ooo`030004003oool0oooo0180oooo00<000@00?ooo`3oool0
2`3oool00`1kOG/0oooo0?ooo`030?ooo`030?l4o`3oool0oooo00<0oooo0P001?l?0?ooo`<00?l0
<@3oool40?l400030?ooo`3o1000o`@006l0oooo000f0?ooo`03000000001000oooo0180oooo00<0
00@00?ooo`3oool02P3oool207]mN`@0oooo00<0o`3o0?ooo`3oool0103oool00`001?l0003o0?oo
o`0<0?ooo`800?l0<03oool01P3o0000o`@00?l0003o1000o`000?l407@0oooo000f0?ooo`030004
003oool0oooo0140oooo00<000@00?ooo`3oool02`3oool207]mN`80oooo0P3o1?l60?ooo`030004
o`3oool0oooo00`0oooo0P00o`0`0?ooo`<0o`@0N03oool003H0oooo00<000000004003oool04@3o
ool00`000000oooo0?ooo`0;0?ooo`0607]mN`3oool0oooo0?ooo`3o0?l0o`Co1@3oool00`001?l0
oooo0?ooo`0;0?ooo`@00?l0;03oool0103o1000o`000?l4003o001k0?ooo`00=P3oool200040180
oooo00<000@00?ooo`3oool02P3oool01@1kOG/0oooo0?ooo`3oool0o`Co00@0oooo0`001?l;0?oo
o`<00?l0:`3oool20?l400030?ooo`3o1000o`@00080o`@0O@3oool003H0oooo00<000000004003o
ool03`3oool00`00000000@00?ooo`0<0?ooo`0407]mN`3oool0oooo0?l0o`D0oooo00<0003o0004
o`3oool02`3oool2003o02/0oooo00@0o`@00?l0003o1000o`00P@3oool001/0oooo0P0010040?oo
o`030004003oool0oooo00<0oooo0P0010030?ooo`@000@01P3oool200040100oooo0P00100<0?oo
o`0507]mN`3oool0oooo0?ooo`3o1?l01@3oool00`001?l0oooo0?ooo`090?ooo`800?l0:P3oool3
0?l408D0oooo000J0?ooo`040000003oool0oooo000400P0oooo00@000000?ooo`3oool000@0103o
ool00`000000oooo0?ooo`050?ooo`03000000001000000000l0oooo00<000000?ooo`3oool02`3o
ool0101kOG/0oooo0?l0o`3o1?l40?ooo`030000o`001?l0oooo00P0oooo0`00o`0W0?ooo`070?l4
003o0000oooo0?l0003o1000o`000?l400260?ooo`006P3oool010001000oooo0?ooo`0010080?oo
o`040004003oool0oooo000400@0oooo00<000@00?ooo`3oool01@3oool00`001000oooo0?ooo`0?
0?ooo`030004003oool0oooo00X0oooo00<0Ngek0?ooo`3oool00P3o1?l20?ooo`8000Co2`3oool2
003o02H0oooo1@3o102:0?ooo`006P3oool010000000oooo0?ooo`0010080?ooo`040000003oool0
oooo000400@0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`0>0?ooo`030004003o
ool0oooo00/0oooo00<0Ngek0?ooo`3o1?l0103oool00`000?l000Co0?ooo`090?ooo`800?l0903o
ool00`3o0000o`@00?l0002?0?ooo`006P3oool010001000oooo0?ooo`0010080?ooo`040004003o
ool0oooo000400@0oooo00<000@00?ooo`3oool01@3oool00`001000oooo0?ooo`0>0?ooo`030004
003oool0oooo00X0oooo00<0Ngek0?ooo`3o1?l0103oool00`001?l0oooo0?ooo`070?ooo`<00?l0
8P3oool60?l40900oooo000J0?ooo`040000003oool0oooo000400P0oooo00@000000?ooo`3oool0
00@00P3oool00`00000000@0000000070?ooo`030000003oool0oooo00d0oooo00<000000?ooo`3o
ool02`3oool00`1kOG/0o`Co0?ooo`030?ooo`030000o`3oool0oooo00L0oooo0`00o`0N0?ooo`07
0?l4003o0000oooo0?l0003o1000o`000?l4002D0?ooo`006`3oool2000400X0oooo0P0010050?oo
o`030004003oool0oooo00D0oooo00<000@00?ooo`3oool03@3oool00`001000oooo0?ooo`090?oo
o`80Ngek00@0o`Co0?ooo`3oool0oooo0P001?l80?ooo`800?l0803oool40?l409P0oooo000f0?oo
o`030000003oool0oooo00`0oooo00<000@00?ooo`3oool02P3oool207]mN`@0oooo00<0003o0004
o`3oool01@3oool2003o01l0oooo00<0o`@00?l0003o1000W03oool003H0oooo00<000@00?ooo`3o
ool02`3oool2000400`0oooo00<0Ngek0?ooo`3oool00P3oool00`001?l0oooo0?ooo`050?ooo`<0
0?l06P3oool20?l400030?ooo`3o1000o`@00080o`@0W@3oool003H0oooo00<000000?ooo`3oool0
2`3oool00`00000000@00?ooo`090?ooo`030?l0o`3o1?l0Ngek0080oooo00<000Co0000o`3oool0
103oool3003o01/0oooo00D0o`000?l4003o0000o`@00?l0002Q0?ooo`00=P3oool00`001000oooo
0?ooo`0;0?ooo`030004003oool0oooo00P0oooo00<0o`Co0?ooo`1kOG/00`001?l60?ooo`<00?l0
6@3oool30?l40:H0oooo000f0?ooo`030000003oool0oooo00X0oooo00<000@00?ooo`3oool0203o
ool01P3o0?l0o`Co07]mN`3oool0003o0004o`@0oooo0`00o`0H0?ooo`060?l4003o0000o`@00?l0
003o1000o`00Y`3oool003H0oooo00<000@00?ooo`3oool0203oool3000400T0oooo0P3o1?l0101k
OG/0oooo0004o`001?l50?ooo`800?l0503oool20?l400030?ooo`3o1000o`@00080o`@0Z`3oool0
03H0oooo00<000000?ooo`3oool0203oool00`00100000000?ooo`070?ooo`030?l4o`3o0?l0oooo
0080Ngek00<0003o0?ooo`3oool00`3oool2003o01D0oooo00@0o`@00?l0003o1000o`00[`3oool0
03H0oooo00<000@00?ooo`3oool0203oool00`001000oooo0?ooo`060?ooo`040?l4o`3oool0oooo
0?ooo`80Ngek103oool2003o01<0oooo103o102c0?ooo`00=P3oool00`000000oooo0?ooo`070?oo
o`030000003oool0oooo00D0oooo00<0o`3o0?l4o`3oool00P3oool00`1kOG/0oooo0?ooo`020?oo
o`800?l04@3oool01P3o1000o`000?ooo`3o0000o`@00?l00;D0oooo000f0?ooo`030004003oool0
oooo00H0oooo00<000@00?ooo`3oool01@3oool30?l4o`030?ooo`001?l0Ngek00@0oooo0`00o`0?
0?ooo`D0o`@0^03oool003H0oooo00<000000?ooo`3oool01@3oool00`000000oooo0?ooo`030?oo
o`080?l0o`3o1?l0o`3o0?ooo`000?l000Co0000o`1kOG/20?ooo`<00?l03`3oool00`3o0000o`@0
0?l0002m0?ooo`00=P3oool00`001000oooo0?ooo`020?ooo`80o`Co00@000@00?ooo`3oool0oooo
1@3o1?l20?ooo`8000Co00@0Ngek0?ooo`3oool0oooo0P00o`0<0?ooo`H0o`@0_P3oool003H0oooo
00<000000?ooo`3oool00P3oool03@00100000000?l4o`3o0?l0o`Co0?l0o`3o1?l0o`3o0?ooo`00
0?l000Co0000o`001?l00P1kOG/20?ooo`800?l02P3oool01P3o0000o`@00?l0003o1000o`000?l4
0<80oooo000f0?ooo`030004003oool0oooo0080oooo00<000@00004o`001?l020001?l20?ooo`04
07]mN`00o`000?l0003o00X0oooo103o10360?ooo`00=P3oool00`00000000@00?ooo`080?ooo`03
0004o`000?l0oooo00800?l00P1kOG/2003o00P0oooo00@0o`000?l4003o0000o`@0bP3oool003H0
oooo00<000@00?ooo`3oool00P3oool2003o00@0oooo1000o`000`1kOG/00?l00?ooo`060?ooo`H0
o`@0c03oool003H0oooo00<000000004003oool00P3oool8003o00<0Ngek1P3oool0103o0000o`@0
0?l0003o103@0?ooo`00=P3oool00`001000oooo0?ooo`080?ooo`80Ngek103oool50?l40=@0oooo
000f0?ooo`03000000001000oooo0080oooo00<0o`@00?l0003oool00`1kOG/02@3o1000o`000?l4
003o0000o`@00?l0003oool0o`000?l4003F0?ooo`00=P3oool2000400<0oooo0`1kOG/50?l40003
0?ooo`3o1000o`@00=X0oooo000f0?ooo`03000000001000oooo0080oooo0P1kOG_S0?ooo`009P3o
ool4000400<0oooo0P0010070?ooo`8000@0j03oool002P0oooo00D000000?ooo`3oool0oooo0000
00020?ooo`030004003oool0oooo00@0oooo00<000000004003oool0i`3oool002P0oooo00D000@0
0?ooo`3oool0oooo000400020?ooo`030004003oool0oooo00@0oooo0`00103W0?ooo`00:03oool0
1@000000oooo0?ooo`3oool000000080oooo00<000@00?ooo`3oool0kP3oool002P0oooo00D000@0
0?ooo`3oool0oooo000400020?ooo`030004003oool0oooo0>h0oooo000V0?ooo`03000000001000
000000<0oooo00@000000?ooo`3oool000@0l03oool002P0oooo00<000@00?ooo`3oool00P3oool2
00040?40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o
0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo003o
0?ooob40oooo003o0?ooob40oooo003o0?ooob40oooo0000\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {0.222812, -15.9116, \
0.0163206, 0.105637}}],

Cell[TextData[{
  "A logarithmic plot of the maximum error for approximating the first \
derivative of the Gaussian ",
  Cell[BoxData[
      \(TraditionalForm\`f(
          x)\  = \ \[ExponentialE]\^\(-\((15\ \((x\  - \ 1/2)\))\)\^2\)\)]],
  "at points on a grid covering the interval [0,1] as a function of the \
number of grid points, n, using machine precision.  Finite differences of \
order 2,4,6, and 8 on a uniform grid are shown in red, green, blue, and \
magenta respectively. Pseudospectral derivatives with uniform (periodic) and \
Chebyshev spacing are shown in black and grey respectively.",
  " "
}], "Caption"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-0.293651 0.263645 0.570493 0.0237705 [
[.23364 .40435 -11 -9.8125 ]
[.23364 .40435 11 0 ]
[.49728 .40435 -14 -9.8125 ]
[.49728 .40435 14 0 ]
[.76093 .40435 -17 -9.8125 ]
[.76093 .40435 17 0 ]
[-0.0125 0 -51.6875 -6.15625 ]
[-0.0125 0 0 6.15625 ]
[-0.0125 .11885 -51.6875 -6.15625 ]
[-0.0125 .11885 0 6.15625 ]
[-0.0125 .23771 -51.6875 -6.15625 ]
[-0.0125 .23771 0 6.15625 ]
[-0.0125 .35656 -47.4375 -6.15625 ]
[-0.0125 .35656 0 6.15625 ]
[-0.0125 .47541 -40 -4.90625 ]
[-0.0125 .47541 0 4.90625 ]
[-0.0125 .59426 -16 -4.90625 ]
[-0.0125 .59426 0 4.90625 ]
[ -0.0005 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.23364 .41685 m
.23364 .4231 L
s
gsave
.23364 .40435 -72 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(100) show
81.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.49728 .41685 m
.49728 .4231 L
s
gsave
.49728 .40435 -75 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1000) show
87.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.76093 .41685 m
.76093 .4231 L
s
gsave
.76093 .40435 -78 -13.8125 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10000) show
93.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
.04936 .41685 m
.04936 .4206 L
s
.09578 .41685 m
.09578 .4206 L
s
.12872 .41685 m
.12872 .4206 L
s
.15427 .41685 m
.15427 .4206 L
s
.17515 .41685 m
.17515 .4206 L
s
.1928 .41685 m
.1928 .4206 L
s
.20809 .41685 m
.20809 .4206 L
s
.22158 .41685 m
.22158 .4206 L
s
.313 .41685 m
.313 .4206 L
s
.35943 .41685 m
.35943 .4206 L
s
.39237 .41685 m
.39237 .4206 L
s
.41792 .41685 m
.41792 .4206 L
s
.4388 .41685 m
.4388 .4206 L
s
.45645 .41685 m
.45645 .4206 L
s
.47173 .41685 m
.47173 .4206 L
s
.48522 .41685 m
.48522 .4206 L
s
.57665 .41685 m
.57665 .4206 L
s
.62308 .41685 m
.62308 .4206 L
s
.65601 .41685 m
.65601 .4206 L
s
.68156 .41685 m
.68156 .4206 L
s
.70244 .41685 m
.70244 .4206 L
s
.72009 .41685 m
.72009 .4206 L
s
.73538 .41685 m
.73538 .4206 L
s
.74887 .41685 m
.74887 .4206 L
s
.84029 .41685 m
.84029 .4206 L
s
.88672 .41685 m
.88672 .4206 L
s
.91966 .41685 m
.91966 .4206 L
s
.94521 .41685 m
.94521 .4206 L
s
.96609 .41685 m
.96609 .4206 L
s
.98374 .41685 m
.98374 .4206 L
s
.99902 .41685 m
.99902 .4206 L
s
.25 Mabswid
0 .41685 m
1 .41685 L
s
0 0 m
.00625 0 L
s
gsave
-0.0125 0 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(24) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .11885 m
.00625 .11885 L
s
gsave
-0.0125 .11885 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(19) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .23771 m
.00625 .23771 L
s
gsave
-0.0125 .23771 -112.688 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(14) show
109.375 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .35656 m
.00625 .35656 L
s
gsave
-0.0125 .35656 -108.438 -10.1562 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20.3125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def
63.000 13.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
%%BeginResource: font Courier-MISO
%%BeginFont: Courier-MISO
/Courier /Courier-MISO MISOfy
%%EndFont
%%EndResource
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(1.) show
75.000 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
76.812 13.750 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(\\264) show
84.625 13.750 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
96.625 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
96.625 9.375 moveto
%%IncludeResource: font Mathematica1Mono
%%IncludeFont: Mathematica1Mono
/Mathematica1Mono findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(-) show
100.875 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(9) show
105.125 9.375 moveto
%%IncludeResource: font Courier-MISO
%%IncludeFont: Courier-MISO
/Courier-MISO findfont 7.125 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .47541 m
.00625 .47541 L
s
gsave
-0.0125 .47541 -101 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(0.0001) show
99.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
0 .59426 m
.00625 .59426 L
s
gsave
-0.0125 .59426 -77 -8.90625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 17.8125 translate 1 -1 scale
/g { setgray} bind def
/k { setcmykcolor} bind def
/p { gsave} bind def
/r { setrgbcolor} bind def
/w { setlinewidth} bind def
/C { curveto} bind def
/F { fill} bind def
/L { lineto} bind def
/rL { rlineto} bind def
/P { grestore} bind def
/s { stroke} bind def
/S { show} bind def
/N {currentpoint 3 -1 roll show moveto} bind def
/Msf { findfont exch scalefont [1 0 0 -1 0 0 ] makefont setfont} bind def
/m { moveto} bind def
/Mr { rmoveto} bind def
/Mx {currentpoint exch pop moveto} bind def
/My {currentpoint pop exch moveto} bind def
/X {0 rmoveto} bind def
/Y {0 exch rmoveto} bind def
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(10) show
75.000 11.250 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.001 w
0 .09617 m
.00375 .09617 L
s
0 .10333 m
.00375 .10333 L
s
0 .10751 m
.00375 .10751 L
s
0 .11048 m
.00375 .11048 L
s
0 .11278 m
.00375 .11278 L
s
0 .11467 m
.00375 .11467 L
s
0 .11626 m
.00375 .11626 L
s
0 .11764 m
.00375 .11764 L
s
0 .21502 m
.00375 .21502 L
s
0 .22218 m
.00375 .22218 L
s
0 .22636 m
.00375 .22636 L
s
0 .22933 m
.00375 .22933 L
s
0 .23164 m
.00375 .23164 L
s
0 .23352 m
.00375 .23352 L
s
0 .23511 m
.00375 .23511 L
s
0 .23649 m
.00375 .23649 L
s
0 .33388 m
.00375 .33388 L
s
0 .34103 m
.00375 .34103 L
s
0 .34522 m
.00375 .34522 L
s
0 .34819 m
.00375 .34819 L
s
0 .35049 m
.00375 .35049 L
s
0 .35237 m
.00375 .35237 L
s
0 .35396 m
.00375 .35396 L
s
0 .35534 m
.00375 .35534 L
s
0 .45273 m
.00375 .45273 L
s
0 .45988 m
.00375 .45988 L
s
0 .46407 m
.00375 .46407 L
s
0 .46704 m
.00375 .46704 L
s
0 .46934 m
.00375 .46934 L
s
0 .47123 m
.00375 .47123 L
s
0 .47282 m
.00375 .47282 L
s
0 .47419 m
.00375 .47419 L
s
0 .57158 m
.00375 .57158 L
s
0 .57874 m
.00375 .57874 L
s
0 .58292 m
.00375 .58292 L
s
0 .58589 m
.00375 .58589 L
s
0 .5882 m
.00375 .5882 L
s
0 .59008 m
.00375 .59008 L
s
0 .59167 m
.00375 .59167 L
s
0 .59305 m
.00375 .59305 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
1 0 0 r
.015 w
.02381 .58459 Mdot
.04936 .58447 Mdot
.06027 .58361 Mdot
.0794 .58127 Mdot
.10317 .57729 Mdot
.12285 .57337 Mdot
.14473 .56985 Mdot
.16309 .56722 Mdot
.18254 .56369 Mdot
.20222 .56018 Mdot
.22158 .55687 Mdot
.24245 .55314 Mdot
.2619 .54955 Mdot
.28158 .54611 Mdot
.30221 .54241 Mdot
.32182 .53888 Mdot
.34127 .53542 Mdot
.36095 .53186 Mdot
.38094 .52827 Mdot
.40065 .52472 Mdot
.42063 .52112 Mdot
.44031 .51757 Mdot
.46031 .51397 Mdot
.48028 .51037 Mdot
.5 .50681 Mdot
.51986 .50323 Mdot
.53967 .49966 Mdot
.55951 .49608 Mdot
.57937 .4925 Mdot
.59923 .48892 Mdot
.61904 .48535 Mdot
.63888 .48177 Mdot
.65873 .47819 Mdot
.67855 .47461 Mdot
.6984 .47104 Mdot
.71824 .46746 Mdot
.7381 .46388 Mdot
.75794 .4603 Mdot
.77779 .45672 Mdot
.79763 .45314 Mdot
.81746 .44957 Mdot
.8373 .44599 Mdot
.85714 .44241 Mdot
.87698 .43883 Mdot
.89683 .43525 Mdot
.91667 .43168 Mdot
.93651 .4281 Mdot
.95635 .42452 Mdot
.97619 .42094 Mdot
0 1 0 r
.02381 .57702 Mdot
.04936 .57674 Mdot
.06027 .57529 Mdot
.0794 .57145 Mdot
.10317 .56496 Mdot
.12285 .55854 Mdot
.14473 .55062 Mdot
.16309 .5446 Mdot
.18254 .53859 Mdot
.20222 .53164 Mdot
.22158 .52472 Mdot
.24245 .51751 Mdot
.2619 .51055 Mdot
.28158 .50346 Mdot
.30221 .49614 Mdot
.32182 .48911 Mdot
.34127 .48209 Mdot
.36095 .47505 Mdot
.38094 .46783 Mdot
.40065 .46074 Mdot
.42063 .45355 Mdot
.44031 .44646 Mdot
.46031 .43925 Mdot
.48028 .43205 Mdot
.5 .42494 Mdot
.51986 .41777 Mdot
.53967 .41063 Mdot
.55951 .40348 Mdot
.57937 .39632 Mdo