(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.0'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     79876,       2542]*)
(*NotebookOutlinePosition[     81149,       2590]*)
(*  CellTagsIndexPosition[     81007,       2581]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Composition and Splitting", "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellTags->"s:1"],

Cell["\<\
In some cases it is useful to split the differential system into \
subsystems and solve each subsystem using appropriate integration methods. \
Recombining the individual solutions often allows certain dynamical \
properties, such as volume, to be conserved.\
\>", "Text"],

Cell[CellGroupData[{

Cell["Definitions", "Subsection"],

Cell[TextData[{
  "We are concerned with initial value problems ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(\ \)\(y' \((t)\) = \ f(y(t))\)\)\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`y(0) = 
        y\_0 \[Element] \[DoubleStruckCapitalR]\^n\)]],
  "given."
}], "Text"],

Cell[CellGroupData[{

Cell["Composition", "Subsubsection"],

Cell["\<\
Composition is a useful device for raising the order of a numerical \
integration scheme.\
\>", "Text"],

Cell["\<\
In contrast to the Aitken-Neville algorithm used in extrapolation, \
composition can conserve geometric properties of the base integration method \
(e.g. symplecticity).\
\>", "Text"],

Cell[TextData[{
  "Let ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\_\(\(f\_, \) \(\[Gamma]\_i\) \
h\)\%\((i)\)\)]],
  " be a basic integration method that takes a step of size ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Gamma]\_i\) h\)]],
  " with ",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_1, \[Ellipsis], \[Gamma]\_s\)]],
  " given real numbers."
}], "Text"],

Cell[TextData[{
  "Then the ",
  Cell[BoxData[
      \(TraditionalForm\`s\)]],
  " stage composition method ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPsi]\_\(f, \ h\)\)]],
  " is given by:"
}], "Text"],

Cell[TextData[Cell[BoxData[
    \(TraditionalForm\`\[CapitalPsi]\_\(f, \ h\) = \ \(\[CapitalPhi]\_\(f, \ \
\(\[Gamma]\_s\) 
              h\)\%\((s)\)\) \[EmptySmallCircle]\ \[CenterEllipsis]\ \
\[EmptySmallCircle]\ \[CapitalPhi]\_\(f, \ \(\[Gamma]\_1\) h\)\%\((1)\)\)]]], \
"Text"],

Cell[TextData[{
  "Often we are interested in composition methods ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPsi]\_\(f, \ h\)\)]],
  " that involve the same base method ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\  = \ \[CapitalPhi]\^\((i)\), \ 
      i\  = 1, \[Ellipsis], \(\(s\)\(.\)\)\)]]
}], "Text"],

Cell[TextData[{
  "An interesting special case is symmetric composition: ",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_i\  = \ \[Gamma]\_\(s - i + 1\)\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`i = 
        1, \[Ellipsis], \[LeftFloor]s/2\[RightFloor]\)]],
  "."
}], "Text"],

Cell["The most common types of composition are:", "Text"],

Cell["Symmetric composition of symmetric second-order methods.", \
"BulletedList"],

Cell[TextData[{
  "Symmetric composition of first-order methods (e.g. a method ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\)]],
  " with its adjoint ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[CapitalPhi]\^*\)\)]],
  ")."
}], "BulletedList"],

Cell["Composition of first-order methods.", "BulletedList"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Splitting", "Subsubsection"],

Cell[TextData[{
  "An ",
  Cell[BoxData[
      \(TraditionalForm\`s\)]],
  "-stage splitting method is a generalization of a composition method in \
which ",
  Cell[BoxData[
      \(TraditionalForm\`f\)]],
  " is broken up in an additive fashion:"
}], "Text"],

Cell[TextData[{
  Cell[BoxData[
      \(TraditionalForm\`f\  = \ f\_1\  + \ \[CenterEllipsis]\  + \ f\_k\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`k\  \[LessEqual] \ s\)]],
  "."
}], "Text"],

Cell[TextData[{
  "The essential point is that there can often be computational advantages in \
solving problems involving ",
  Cell[BoxData[
      \(TraditionalForm\`f\_i\)]],
  " instead of ",
  Cell[BoxData[
      \(TraditionalForm\`f\)]],
  "."
}], "Text"],

Cell[TextData[{
  "An ",
  Cell[BoxData[
      \(TraditionalForm\`s\)]],
  " stage splitting method is a composition of the form:"
}], "Text"],

Cell[TextData[Cell[BoxData[
    \(TraditionalForm\`\[CapitalPsi]\_\(f, \ h\) = \ \(\[CapitalPhi]\_\(f\_s, \
\(\[Gamma]\_s\) 
              h\)\%\((s)\)\) \[EmptySmallCircle]\ \[CenterEllipsis]\ \
\[EmptySmallCircle]\ \[CapitalPhi]\_\(f\_1, \ \(\[Gamma]\_1\) \
h\)\%\((1)\)\)]]], "Text"],

Cell[TextData[{
  "with ",
  Cell[BoxData[
      \(TraditionalForm\`f\_1, \[Ellipsis], f\_s\)]],
  " not necessarily distinct."
}], "Text"],

Cell["\<\
Each base integration method now only solves part of the problem, \
but a suitable composition can still give rise to a numerical scheme with \
advantageous properties.\
\>", "Text"],

Cell[TextData[{
  "If the vector field ",
  Cell[BoxData[
      \(TraditionalForm\`f\_i\)]],
  " is integrable, then the exact solution or flow ",
  Cell[BoxData[
      \(TraditionalForm\`\[CurlyPhi]\_\(f\_i, \ h\)\)]],
  " can be used in place of a numerical integration method."
}], "Text"],

Cell["\<\
A splitting method may also use a mixture of flows and numerical \
methods.\
\>", "Text"],

Cell[TextData[{
  "An example is Lie-Trotter splitting [",
  ButtonBox["T59",
    ButtonData:>{"References.nb", "T59"},
    ButtonStyle->"Hyperlink"],
  "]:\nSplit ",
  Cell[BoxData[
      \(TraditionalForm\`f\  = \ f\_1 + f\_2\)]],
  " with ",
  Cell[BoxData[
      \(TraditionalForm\`\[Gamma]\_1 = \ \(\[Gamma]\_2\  = \ 1\)\)]],
  "; then ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPsi]\_\(f, \ h\) = \ \(\[CurlyPhi]\_\(f\_2, 
              h\)\%\((2)\)\) \[EmptySmallCircle]\ \[CurlyPhi]\_\(f\_1, \ \
h\)\%\((1)\)\)]],
  " yields a first-order integration method."
}], "Text"],

Cell["\<\
Computationally it can be advantageous to combine flows using the \
group property:\
\>", "Text"],

Cell[TextData[{
  Cell[BoxData[
      \(TraditionalForm\`\[CurlyPhi]\_\(f\_i, \ h\_1 + h\_2\) = \ \[CurlyPhi]\
\_\(f\_i, \ h\_2\)\ \[EmptySmallCircle]\ \[CurlyPhi]\_\(f\_i, \ h\_1\)\)]],
  "."
}], "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Implementation", "Subsection"],

Cell[TextData[{
  "Several changes to the new ",
  StyleBox["NDSolve", "MR"],
  " framework were needed in order to implement splitting and composition \
methods."
}], "Text"],

Cell["Allow a method to call an arbitrary number of submethods.", \
"BulletedList"],

Cell["\<\
The ability to pass around a function for numerically evaluating a \
subfield, instead of the entire vector field.\
\>", "BulletedList"],

Cell[TextData[{
  "A ",
  StyleBox["LocallyExact", "MR"],
  " method to compute the flow; analytically solve a subsystem and advance \
the (local) solution numerically."
}], "BulletedList"],

Cell["\<\
Cache data for identical methods to avoid repeated initialization. \
Data for numerically evaluating identical subfields is also cached.\
\>", \
"BulletedList"],

Cell["\<\
A simplified input syntax allows omitted vector fields and methods \
to be filled in cyclically. These must be defined unambiguously:\
\>", "Text"],

Cell[TextData[{
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            RowBox[{
              FormBox[\({\(f\_1\)\(,\)\),
                "TraditionalForm"], \(f\_2\)}], ",", \(f\_1\), 
            ",", \(f\_2\)}], "}"}], TraditionalForm]]],
  " can be input as ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            FormBox[\({\(f\_1\)\(,\)\),
              "TraditionalForm"], \(f\_2\)}], "}"}], TraditionalForm]]],
  "."
}], "Text"],

Cell[TextData[{
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            RowBox[{
              FormBox[\({\(f\_1\)\(,\)\),
                "TraditionalForm"], \(f\_2\)}], ",", \(f\_3\), ",", \(f\_2\), 
            ",", \(f\_1\)}], "}"}], TraditionalForm]]],
  "cannot be input as ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            RowBox[{
              FormBox[\({\(f\_1\)\(,\)\),
                "TraditionalForm"], \(f\_2\)}], ",", \(f\_3\)}], "}"}], 
        TraditionalForm]]],
  " since this corresponds to ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          RowBox[{
            RowBox[{
              FormBox[\({\(f\_1\)\(,\)\),
                "TraditionalForm"], \(f\_2\)}], ",", \(f\_3\), ",", \(f\_1\), 
            ",", \(f\_2\)}], "}"}], TraditionalForm]]],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell["Nested methods", "Subsubsection"],

Cell["\<\
The following example constructs a high-order splitting method from \
a low-order splitting using composition.\
\>", "Text"],

Cell[TextData[{
  Cell[BoxData[
      FormBox[
        FrameBox[
          StyleBox["NDSolve",
            "MR"],
          BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], TraditionalForm]]],
  " ",
  Cell[BoxData[
      FormBox[GridBox[{
            {"\[LongRightArrow]", 
              RowBox[{
                FrameBox[
                  StyleBox["Composition",
                    "MR"],
                  BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", 
                FormBox[GridBox[{
                      {"\[UpperRightArrow]", 
                        RowBox[{
                          FrameBox[
                            RowBox[{
                              RowBox[{
                                StyleBox["Splitting",
                                  "MR"], 
                                StyleBox[" ",
                                  "Text"], 
                                StyleBox["f",
                                  "Text"]}], 
                              StyleBox["=",
                                "Text"], 
                              RowBox[{
                                SubscriptBox[
                                  StyleBox["f",
                                    "Text"], "1"], "+", \(f\_2\)}]}],
                            BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", " ", 
                          FormBox[GridBox[{
                                {"\[UpperRightArrow]", 
                                  FrameBox[
                                    RowBox[{
                                      StyleBox["LocallyExact",
                                        "MR"], "  ", \(f\_1\)}],
                                    BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
                                {"\[LongRightArrow]", GridBox[{
                                      {
                                        FrameBox[
                                        RowBox[{
                                        StyleBox["ImplicitMidpoint",
                                        "MR"], " ", \(f\_2\)}],
                                        BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}],
                                         ""}
                                      }]},
                                {"\[LowerRightArrow]", GridBox[{
                                      {
                                        FrameBox[
                                        RowBox[{
                                        StyleBox["LocallyExact",
                                        "MR"], " ", \(f\_1\)}],
                                        BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}],
                                         ""}
                                      }]}
                                }],
                            "TraditionalForm"]}]},
                      {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
                      {"\[LongRightArrow]", 
                        RowBox[{
                          FrameBox[
                            RowBox[{
                              RowBox[{
                                StyleBox["Splitting",
                                  "MR"], 
                                StyleBox[" ",
                                  "Text"], 
                                StyleBox["f",
                                  "Text"]}], 
                              StyleBox["=",
                                "Text"], 
                              RowBox[{
                                SubscriptBox[
                                  StyleBox["f",
                                    "Text"], "1"], "+", \(f\_2\)}]}],
                            BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", " ", 
                          FormBox[GridBox[{
                                {"\[UpperRightArrow]", 
                                  FrameBox[
                                    RowBox[{
                                      StyleBox["LocallyExact",
                                        "MR"], "  ", \(f\_1\)}],
                                    BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
                                {"\[LongRightArrow]", GridBox[{
                                      {
                                        FrameBox[
                                        RowBox[{
                                        StyleBox["ImplicitMidpoint",
                                        "MR"], " ", \(f\_2\)}],
                                        BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}],
                                         ""}
                                      }]},
                                {"\[LowerRightArrow]", GridBox[{
                                      {
                                        FrameBox[
                                        RowBox[{
                                        StyleBox["LocallyExact",
                                        "MR"], " ", \(f\_1\)}],
                                        BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}],
                                         ""}
                                      }]}
                                }],
                            "TraditionalForm"]}]},
                      {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
                      {"\[LowerRightArrow]", 
                        RowBox[{
                          FrameBox[
                            RowBox[{
                              RowBox[{
                                StyleBox["Splitting",
                                  "MR"], 
                                StyleBox[" ",
                                  "Text"], 
                                StyleBox["f",
                                  "Text"]}], 
                              StyleBox["=",
                                "Text"], 
                              RowBox[{
                                SubscriptBox[
                                  StyleBox["f",
                                    "Text"], "1"], "+", \(f\_2\)}]}],
                            BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], " ", " ", 
                          FormBox[GridBox[{
                                {"\[UpperRightArrow]", 
                                  FrameBox[
                                    RowBox[{
                                      StyleBox["LocallyExact",
                                        "MR"], "  ", \(f\_1\)}],
                                    BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
                                {"\[LongRightArrow]", GridBox[{
                                      {
                                        FrameBox[
                                        RowBox[{
                                        StyleBox["ImplicitMidpoint",
                                        "MR"], " ", \(f\_2\)}],
                                        BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}],
                                         ""}
                                      }]},
                                {"\[LowerRightArrow]", GridBox[{
                                      {
                                        FrameBox[
                                        RowBox[{
                                        StyleBox["LocallyExact",
                                        "MR"], " ", \(f\_1\)}],
                                        BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}],
                                         ""}
                                      }]}
                                }],
                            "TraditionalForm"]}]}
                      }],
                  "TraditionalForm"]}]}
            }], TraditionalForm]]]
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simplification", "Subsubsection"],

Cell[TextData[{
  "A more efficient integrator can be obtained in the previous example using \
the group property of flows and calling the ",
  StyleBox["Splitting", "MR"],
  " method directly."
}], "Text"],

Cell[TextData[{
  Cell[BoxData[
      FormBox[
        FrameBox[
          StyleBox["NDSolve",
            "MR"],
          BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], TraditionalForm]]],
  " ",
  Cell[BoxData[
      FormBox[GridBox[{
            {"\[LongRightArrow]", 
              RowBox[{
                FrameBox[
                  RowBox[{
                    RowBox[{
                      StyleBox["Splitting",
                        "MR"], 
                      StyleBox[" ",
                        "Text"], 
                      StyleBox["f",
                        "Text"]}], 
                    StyleBox["=",
                      "Text"], 
                    RowBox[{
                      SubscriptBox[
                        StyleBox["f",
                          "Text"], "1"], "+", \(f\_2\)}]}],
                  BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], 
                FormBox[GridBox[{
                      {"\[UpperRightArrow]", 
                        RowBox[{GridBox[{
                              {
                                RowBox[{
                                  FrameBox[
                                    RowBox[{
                                      StyleBox["LocallyExact",
                                        "MR"], "  ", \(f\_1\)}],
                                    BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], 
                                  " "}]},
                              {
                                FrameBox[
                                  RowBox[{
                                    StyleBox["ImplicitMidpoint",
                                      "MR"], " ", \(f\_2\)}],
                                  BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}]}
                              }], 
                          FormBox["",
                            "TraditionalForm"]}]},
                      {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
                      {"\[LongRightArrow]", 
                        RowBox[{GridBox[{
                              {
                                RowBox[{
                                  FrameBox[
                                    RowBox[{
                                      StyleBox["LocallyExact",
                                        "MR"], "  ", \(f\_1\)}],
                                    BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], 
                                  " "}]},
                              {
                                FrameBox[
                                  RowBox[{
                                    StyleBox["ImplicitMidpoint",
                                      "MR"], " ", \(f\_2\)}],
                                  BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
                              {
                                RowBox[{
                                  FrameBox[
                                    RowBox[{
                                      StyleBox["LocallyExact",
                                        "MR"], "  ", \(f\_1\)}],
                                    BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], 
                                  " "}]}
                              }], " ", 
                          FormBox["",
                            "TraditionalForm"]}]},
                      {"\[VerticalEllipsis]", "\[VerticalEllipsis]"},
                      {"\[LowerRightArrow]", 
                        RowBox[{GridBox[{
                              {
                                FrameBox[
                                  RowBox[{
                                    StyleBox["ImplicitMidpoint",
                                      "MR"], " ", \(f\_2\)}],
                                  BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}]},
                              {
                                RowBox[{
                                  FrameBox[
                                    RowBox[{
                                      StyleBox["LocallyExact",
                                        "MR"], "  ", \(f\_1\)}],
                                    BoxMargins->{{0.2, 0.2}, {0.4, 0.4}}], 
                                  " "}]}
                              }], "  "}]}
                      }],
                  "TraditionalForm"]}]}
            }], TraditionalForm]]]
}], "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Examples", "Section",
  CellTags->"s:2"],

Cell[TextData[{
  "The following examples will use a second-order symmetric splitting known \
as the Strang splitting [",
  ButtonBox["S68",
    ButtonData:>{"References.nb", "S68"},
    ButtonStyle->"Hyperlink"],
  "], [",
  ButtonBox["M68",
    ButtonData:>{"References.nb", "M68"},
    ButtonStyle->"Hyperlink"],
  "]. The following defines a function to evaluate the coefficients to a \
given precision."
}], "Text"],

Cell[BoxData[
    \(StrangSplittingCoefficients[2, \ p_] := \ 
      N[{1/2, 1, 1/2}, \ p]; \)], "Input",
  CellLabel->"In[1]:=",
  InitializationCell->True],

Cell[TextData[{
  "This defines a method known as symplectic leapfrog in terms of the method \
",
  StyleBox["SymplecticPartitionedRungeKutta", "MR"],
  "."
}], "Text"],

Cell[BoxData[
    \(SymplecticLeapfrog\  = \ {"\<SymplecticPartitionedRungeKutta\>", \ \
"\<DifferenceOrder\>" \[Rule] 2, \ "\<PositionVariables\>" :> 
          qvars}; \)], "Input",
  CellLabel->"In[2]:=",
  InitializationCell->True],

Cell["Load a package with some useful example problems.", "Text"],

Cell[BoxData[
    \(Needs["\<DifferentialEquations`NDSolveProblems`\>"]; \)], "Input",
  CellLabel->"In[3]:=",
  InitializationCell->True],

Cell[CellGroupData[{

Cell["Symplectic splitting", "Subsection"],

Cell[CellGroupData[{

Cell["Symplectic leapfrog", "Subsubsection"],

Cell[TextData[{
  StyleBox["SymplecticPartitionedRungeKutta", "MR"],
  " is an efficient method for solving separable Hamiltonian systems ",
  Cell[BoxData[
      \(TraditionalForm\`H(p, \ q)\  = \ T(p)\  + \ V(q)\)]],
  " with favorable long time dynamics."
}], "Text"],

Cell[TextData[{
  StyleBox["Splitting", "MR"],
  " is a more general-purpose method, but it can be used to construct \
partitioned symplectic methods (though it is somewhat less efficient than ",
  StyleBox["SymplecticPartitionedRungeKutta", "MR"],
  ")."
}], "Text"],

Cell[TextData[{
  "Consider the harmonic oscillator that arises from a linear differential \
system that is governed by the separable Hamiltonian ",
  StyleBox["H",
    FontSlant->"Italic"],
  "(",
  StyleBox["p",
    FontSlant->"Italic"],
  ",",
  StyleBox["q",
    FontSlant->"Italic"],
  ") = ",
  Cell[BoxData[
      \(TraditionalForm\`p\^2/2\)]],
  "+ ",
  Cell[BoxData[
      \(TraditionalForm\`q\^2/2\)]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(system\  = \ GetNDSolveProblem["\<HarmonicOscillator\>"]\)], "Input",
  CellLabel->"In[20]:="],

Cell[BoxData[
    RowBox[{"NDSolveProblem", "[", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{"{", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_1\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(Y\_2[T]\)}], ",", 
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_2\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(-Y\_1[T]\)}]}], "}"}], 
          ",", \({Y\_1[0] \[Equal] 1, Y\_2[0] \[Equal] 0}\), 
          ",", \({Y\_1[T], Y\_2[T]}\), ",", \({T, 0, 10}\), ",", \({}\), 
          ",", \({1\/2\ \((Y\_1[T]\^2 + Y\_2[T]\^2)\)}\)}], "}"}], 
      "]"}]], "Output",
  CellLabel->"Out[20]="]
}, Open  ]],

Cell["\<\
Split the Hamiltonian vector field into independent components \
governing momentum and position. This is done by setting the relevant \
right-hand sides of the equations to zero.\
\>", "Text"],

Cell[BoxData[{
    \(eqs\  = \ system["\<System\>"[]]; \), "\[IndentingNewLine]", 
    \(Y1\  = \ eqs; \), "\[IndentingNewLine]", 
    \(Part[Y1, \ 1, 2]\  = \ 0; \), "\[IndentingNewLine]", 
    \(Y2\  = \ eqs; \), "\[IndentingNewLine]", 
    \(Part[Y2, \ 2, 2]\  = \ 0; \)}], "Input",
  CellLabel->"In[21]:="],

Cell["\<\
This composition of weighted (first-order) Euler integration steps \
correspond to the symplectic (second-order) leapfrog method.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(tfinal\  = \ 1; \), "\[IndentingNewLine]", 
    \(time\  = \ {T, \ 0, \ tfinal}; \), "\[IndentingNewLine]", 
    \(qvars\  = \ {\(Subscript[Y, 1]\)[T]}; \), "\[IndentingNewLine]", 
    \(splittingsol\  = \ 
      NDSolve[system, \ time, \ 
        StartingStepSize \[Rule] 1/10, \[IndentingNewLine]\ 
        Method \[Rule] {"\<Splitting\>", \ "\<Coefficients\>" \[Rule] 
              StrangSplittingCoefficients, \ "\<DifferenceOrder\>" \[Rule] 
              2, \ "\<Equations\>" \[Rule] {Y1, Y2, 
                Y1}, \[IndentingNewLine]"\<Method\>" \[Rule] \
{"\<ExplicitEuler\>", \ "\<ExplicitEuler\>", \ "\<ExplicitEuler\>"}}]\)}], \
"Input",
  CellLabel->"In[28]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[31]="]
}, Open  ]],

Cell[TextData[{
  "The method ",
  StyleBox["ExplicitEuler", "MR"],
  " could only have been specified once, since the second and third instances \
would have been filled in cyclically."
}], "Text"],

Cell["This is the result at the end of the integration step.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[splittingsol\  /. \ T \[Rule] tfinal]\)], "Input",
  CellLabel->"In[32]:="],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
  CellLabel->"Out[32]//InputForm="]
}, Open  ]],

Cell["\<\
This invokes the built-in integration method corresponding to the \
symplectic leapfrog integrator.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sprksol\  = \ 
      NDSolve[system, \ time, \ StartingStepSize \[Rule] 1/10, \ 
        Method \[Rule] SymplecticLeapfrog]\)], "Input",
  CellLabel->"In[33]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[33]="]
}, Open  ]],

Cell["\<\
The result at the end of the integration step is identical to the \
result of the splitting method.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[sprksol\  /. \ T \[Rule] tfinal]\)], "Input",
  CellLabel->"In[34]:="],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
  CellLabel->"Out[34]//InputForm="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Composition of symplectic leapfrog", "Subsubsection"],

Cell[TextData[{
  "This takes the symplectic leapfrog scheme as the base integration method \
and constructs a fourth-order symplectic integrator using a symmetric \
composition of Yoshida [",
  ButtonBox["Y90",
    ButtonData:>{"References.nb", "Y90"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(YoshidaCoefficients\  = \ 
      RootReduce[{1/\((2 - 2^\((1/3)\))\), \(-2^\((1/3)\)\)/\((2 - 
                2^\((1/3)\))\), 
          1/\((2 - 
                2^\((1/3)\))\)}]; \[IndentingNewLine]\), \
"\[IndentingNewLine]", 
    \(YoshidaCompositionCoefficients[4, \ p_] := \ 
      N[YoshidaCoefficients, \ 
        p]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(splittingsol\  = \ 
      NDSolve[system, time, \ 
        StartingStepSize \[Rule] 1/10, \[IndentingNewLine]\ 
        Method \[Rule] {"\<Composition\>", \ "\<Coefficients\>" \[Rule] 
              YoshidaCompositionCoefficients, \
\[IndentingNewLine]"\<DifferenceOrder\>" \[Rule] 
              4, \ "\<Method\>" \[Rule] {SymplecticLeapfrog}}]\)}], "Input",
  CellLabel->"In[43]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[45]="]
}, Open  ]],

Cell["This is the result at the end of the integration step.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[splittingsol\  /. \ T \[Rule] tfinal]\)], "Input",
  CellLabel->"In[42]:="],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5403078808898406, Subscript[Y, 2][1] -> \
-0.8414706295697821}}\
\>", "Output",
  CellLabel->"Out[42]//InputForm="]
}, Open  ]],

Cell["\<\
This invokes the built-in symplectic integration method using \
coefficients for the fourth-order methods of Ruth, Yoshida.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(SPRK4[4, prec_] := 
      N[{{Root[\(-1\) + 12*#1 - 48*#1^2 + 48*#1^3 &, 1, 0], 
            Root[1 - 24*#1^2 + 48*#1^3 &, 1, 0], 
            Root[1 - 24*#1^2 + 48*#1^3 &, 1, 0], 
            Root[\(-1\) + 12*#1 - 48*#1^2 + 48*#1^3 &, 1, 
              0]}, {Root[\(-1\) + 6*#1 - 12*#1^2 + 6*#1^3 &, 1, 0], 
            Root[1 - 3*#1 + 3*#1^2 + 3*#1^3 &, 1, 0], 
            Root[\(-1\) + 6*#1 - 12*#1^2 + 6*#1^3 &, 1, 0], 0}}, 
        prec]; \[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(\ sprksol\  = \ 
      NDSolve[system, time, \ 
        StartingStepSize \[Rule] 
          1/10, \ \[IndentingNewLine]Method \[Rule] \
{"\<SymplecticPartitionedRungeKutta\>", \ "\<Coefficients\>" \[Rule] 
              SPRK4, "\<DifferenceOrder\>" \[Rule] 
              4, "\<PositionVariables\>" \[Rule] qvars}]\)}], "Input",
  CellLabel->"In[46]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 1.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[47]="]
}, Open  ]],

Cell["\<\
The result at the end of the integration step is identical to the \
result of the composition method.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[sprksol\  /. \ T \[Rule] tfinal]\)], "Input",
  CellLabel->"In[49]:="],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5403078808898406, Subscript[Y, 2][1] -> \
-0.8414706295697821}}\
\>", "Output",
  CellLabel->"Out[49]//InputForm="]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Hybrid methods", "Subsection"],

Cell["\<\
While a closed-form solution often does not exist for the entire \
vector field, in some cases it is possible to analytically solve a system of \
differential equations for part of the vector field.\
\>", "Text"],

Cell[TextData[{
  "When a solution can be found by ",
  StyleBox["DSolve", "MR"],
  ", direct numerical evaluation can be used to locally advance the \
solution."
}], "Text"],

Cell[TextData[{
  "This idea is implemented in the method ",
  StyleBox[ButtonBox["LocallyExact",
    ButtonData:>{"LocallyExact.nb", None},
    ButtonStyle->"Hyperlink"], "MR"],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell["Harmonic oscillator test example", "Subsubsection"],

Cell["\<\
This example checks that the solution for the exact flows of split \
components of the harmonic oscillator equations is the same as applying \
Euler's method to each of the split components.\
\>", "Text"],

Cell[BoxData[{
    \(system\  = \ 
      GetNDSolveProblem["\<HarmonicOscillator\>"]; \), "\[IndentingNewLine]", \

    \(eqs\  = \ system["\<System\>"[]]; \), "\[IndentingNewLine]", 
    \(Y1\  = \ eqs; \), "\[IndentingNewLine]", 
    \(Part[Y1, \ 1, 2]\  = \ 0; \), "\[IndentingNewLine]", 
    \(Y2\  = \ eqs; \), "\[IndentingNewLine]", 
    \(Part[Y2, \ 2, 2]\  = \ 0; \), "\[IndentingNewLine]", 
    \(tfinal\  = \ 1; \), "\[IndentingNewLine]", 
    \(time\  = \ {T, \ 0, \ tfinal}; \)}], "Input",
  CellLabel->"In[98]:="],

Cell[BoxData[
    \(solexact\  = \ 
      NDSolve[system, \ time, \ 
        StartingStepSize \[Rule] 1/10, \[IndentingNewLine]\ 
        Method \[Rule] {NDSolve`Splitting, \ "\<Coefficients\>" \[Rule] 
              StrangSplittingCoefficients, \
\[IndentingNewLine]"\<DifferenceOrder\>" \[Rule] 
              2, \ "\<Equations\>" \[Rule] {Y1, Y2, 
                Y1}, \ "\<Method\>" \[Rule] {"\<LocallyExact\>"}}]; \)], \
"Input",
  CellLabel->"In[106]:="],

Cell[CellGroupData[{

Cell[BoxData[
    \(InputForm[solexact\  /. \ T \[Rule] 1]\)], "Input",
  CellLabel->"In[107]:="],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
  CellLabel->"Out[107]//InputForm="]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
    \(soleuler\  = \ 
      NDSolve[system, \ time, \ 
        StartingStepSize \[Rule] 1/10, \[IndentingNewLine]\ 
        Method \[Rule] {NDSolve`Splitting, \ "\<Coefficients\>" \[Rule] 
              StrangSplittingCoefficients, \
\[IndentingNewLine]"\<DifferenceOrder\>" \[Rule] 
              2, \ "\<Equations\>" \[Rule] {Y1, Y2, 
                Y1}, \ "\<Method\>" \[Rule] {"\<ExplicitEuler\>"}}]; \
\[IndentingNewLine]\), "\[IndentingNewLine]", 
    \(InputForm[soleuler\  /. \ T \[Rule] tfinal]\)}], "Input",
  CellLabel->"In[108]:="],

Cell["\<\
{{Subscript[Y, 1][1] -> 0.5399512509335085, Subscript[Y, 2][1] -> \
-0.8406435124348495}}\
\>", "Output",
  CellLabel->"Out[109]//InputForm="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Hybrid numeric-symbolic splitting methods (ABC flow)", "Subsubsection"],

Cell["\<\
Consider the Arnold, Beltrami, and Childress flow, a widely studied \
model for volume-preserving three-dimensional flows.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(system\  = \ 
      GetNDSolveProblem["\<ArnoldBeltramiChildress\>"]\)], "Input",
  CellLabel->"In[70]:="],

Cell[BoxData[
    RowBox[{"NDSolveProblem", "[", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{"{", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_1\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(3\/4\ Cos[Y\_2[T]] + Sin[Y\_3[T]]\)}], ",", 
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_2\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(Cos[Y\_3[T]] + Sin[Y\_1[T]]\)}], ",", 
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_3\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(Cos[Y\_1[T]] + 3\/4\ Sin[Y\_2[T]]\)}]}], "}"}], 
          ",", \({Y\_1[0] \[Equal] 1\/4, Y\_2[0] \[Equal] 1\/3, 
            Y\_3[0] \[Equal] 1\/2}\), ",", \({Y\_1[T], Y\_2[T], Y\_3[T]}\), 
          ",", \({T, 0, 100}\), ",", \({}\), ",", \({}\)}], "}"}], 
      "]"}]], "Output",
  CellLabel->"Out[70]="]
}, Open  ]],

Cell["\<\
When applied directly, a volume preserving integrator would not in \
general preserve symmetries. A symmetry-preserving integrator, such as the \
implicit midpoint rule, would not preserve volume.\
\>", "Text"],

Cell["\<\
This defines a splitting of the system by setting some of the right \
hand side components to zero.\
\>", "Text"],

Cell[BoxData[{
    \(eqs\  = \ system["\<System\>"[]]; \), "\[IndentingNewLine]", 
    \(Y1\  = \ eqs; \), "\[IndentingNewLine]", 
    \(Part[Y1, \ 2, 2]\  = \ 0; \), "\[IndentingNewLine]", 
    \(Y2\  = \ eqs; \), "\[IndentingNewLine]", 
    \(Part[Y2, \ {1, 3}, 2]\  = \ 0; \)}], "Input",
  CellLabel->"In[71]:="],

Cell[CellGroupData[{

Cell[BoxData[
    \(Y1\)], "Input",
  CellLabel->"In[76]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_1\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], 
          "\[Equal]", \(3\/4\ Cos[Y\_2[T]] + Sin[Y\_3[T]]\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_2\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_3\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], 
          "\[Equal]", \(Cos[Y\_1[T]] + 3\/4\ Sin[Y\_2[T]]\)}]}], 
      "}"}]], "Output",
  CellLabel->"Out[76]="]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(Y2\)], "Input",
  CellLabel->"In[77]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_1\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_2\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], 
          "\[Equal]", \(Cos[Y\_3[T]] + Sin[Y\_1[T]]\)}], ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox[\(Y\_3\), "\[Prime]",
              MultilineFunction->None], "[", "T", "]"}], "\[Equal]", "0"}]}], 
      "}"}]], "Output",
  CellLabel->"Out[77]="]
}, Open  ]],

Cell[TextData[{
  "The system for ",
  StyleBox["Y1", "MR"],
  " is solvable exactly by ",
  StyleBox["DSolve", "MR"],
  " so that we can use the ",
  StyleBox["LocallyExact", "MR"],
  " method."
}], "Text"],

Cell[TextData[{
  StyleBox["Y2", "MR"],
  " is not solvable, however, so we need to use a suitable numerical \
integrator in order to obtain the desired properties in the splitting \
method."
}], "Text"],

Cell[TextData[{
  "This defines a method for computing the implicit midpoint rule in terms of \
the built-in ",
  StyleBox["ImplicitRungeKutta", "MR"],
  " method."
}], "Text"],

Cell[BoxData[
    \(\(ImplicitMidpoint\  = {"\<FixedStep\>", 
          Method \[Rule] \ {"\<ImplicitRungeKutta\>", \ "\<Coefficients\>" \
\[Rule] "\<ImplicitRungeKuttaGaussCoefficients\>", "\<DifferenceOrder\>" \
\[Rule] 2, \ ImplicitSolver \[Rule] {FixedPoint, 
                  AccuracyGoal \[Rule] MachinePrecision, \ 
                  PrecisionGoal \[Rule] 
                    MachinePrecision, "\<IterationSafetyFactor\>" \[Rule] 
                    1}}};\)\)], "Input",
  CellLabel->"In[78]:="],

Cell[TextData[{
  "This defines a second-order, volume-preserving, reversing symmetry-group \
integrator [",
  ButtonBox["M02",
    ButtonData:>{"References.nb", "M02"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(splittingsol\  = \ 
      NDSolve[system, \[IndentingNewLine]StartingStepSize \[Rule] 
          1/10, \[IndentingNewLine]\ 
        Method \[Rule] {"\<Splitting\>", \[IndentingNewLine]\ "\<Coefficients\
\>" \[Rule] StrangSplittingCoefficients, \ "\<DifferenceOrder\>" \[Rule] 
              2, \[IndentingNewLine]"\<Equations\>" \[Rule] {Y2, Y1, 
                Y2}, \ "\<Method\>" \[Rule] {"\<LocallyExact\>", 
                ImplicitMidpoint, "\<LocallyExact\>"}}]\)], "Input",
  CellLabel->"In[97]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_3[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 100.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[97]="]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Lotka-Volterra equations", "Subsubsection"],

Cell[TextData[{
  "Various numerical integrators for this system are compared within ",
  ButtonBox["numerical methods for solving the Lotka-Volterra equations",
    ButtonData:>{"LotkaVolterra.nb", None},
    ButtonStyle->"Hyperlink"],
  "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Euler's equations", "Subsubsection"],

Cell[TextData[{
  "Various numerical integrators for Euler's equations are compared within ",
  ButtonBox["Rigid body solvers",
    ButtonData:>{"RigidBody.nb", None},
    ButtonStyle->"Hyperlink"],
  "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Non-autonomous vector fields", "Subsubsection"],

Cell["\<\
Consider the Duffing oscillator, a forced planar non-autonomous \
differential system.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(system\  = \ GetNDSolveProblem["\<DuffingOscillator\>"]\)], "Input",
  CellLabel->"In[4]:="],

Cell[BoxData[
    RowBox[{"NDSolveProblem", "[", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{"{", 
            RowBox[{
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_1\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(Y\_2[T]\)}], ",", 
              RowBox[{
                RowBox[{
                  SuperscriptBox[\(Y\_2\), "\[Prime]",
                    MultilineFunction->None], "[", "T", "]"}], 
                "\[Equal]", \(\(3\ Cos[T]\)\/10 + Y\_1[T] - Y\_1[T]\^3 + 
                  Y\_2[T]\/4\)}]}], "}"}], 
          ",", \({Y\_1[0] \[Equal] 0, Y\_2[0] \[Equal] 1}\), 
          ",", \({Y\_1[T], Y\_2[T]}\), ",", \({T, 0, 10}\), ",", \({}\), 
          ",", \({}\)}], "}"}], "]"}]], "Output",
  CellLabel->"Out[4]="]
}, Open  ]],

Cell["This defines a splitting of the system.", "Text"],

Cell[BoxData[{
    RowBox[{
      RowBox[{"Y1", " ", "=", " ", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_1\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(Y\_2[T]\)}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_2\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(Y\_2[T]\/4\)}]}], "}"}]}], 
      ";"}], "\[IndentingNewLine]", 
    RowBox[{
      RowBox[{"Y2", " ", "=", " ", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_1\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
              "0"}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_2\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(\(3\ Cos[T]\)\/10 + Y\_1[T] - Y\_1[T]\^3\)}]}], 
          "}"}]}], ";"}]}], "Input",
  CellLabel->"In[5]:="],

Cell["\<\
The splitting of the time component among the vector fields is \
ambiguous, so the method issues an error message.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(splittingsol\  = \ 
      NDSolve[system, \ StartingStepSize \[Rule] 1/10, \[IndentingNewLine]\ 
        Method \[Rule] {"\<Splitting\>", \[IndentingNewLine]\ "\<Coefficients\
\>" \[Rule] StrangSplittingCoefficients, "\<DifferenceOrder\>" \[Rule] 
              2, \[IndentingNewLine]"\<Equations\>" \[Rule] {Y2, Y1, 
                Y1}, "\<Method\>" \[Rule] {"\<LocallyExact\>"}}]\)], "Input",
  CellLabel->"In[7]:="],

Cell[BoxData[
    RowBox[{\(NDSolve::"spltdep"\), \(\(:\)\(\ \)\), "\<\"The differential \
system \\!\\({0, \\(\\(\\(3\\\\ \\(\\(Cos[T]\\)\\)\\)\\/10 + \
\\(\\(Y\\_1[T]\\)\\) - \\(Y\\_1[T]\\)\\^3\\)\\)}\\) in the method \
\\!\\(\\\"Splitting\\\"\\) depends on \\!\\(T\\) which is ambiguous. The \
differential system should be in autonomous form. \\!\\(\\*ButtonBox[\\\"More\
\[Ellipsis]\\\", ButtonStyle->\\\"RefGuideLinkText\\\", ButtonFrame->None, \
ButtonData:>\\\"NDSolve::spltdep\\\"]\\)\"\>"}]], "Message",
  CellLabel->"From In[7]:="],

Cell[BoxData[
    RowBox[{\(NDSolve::"initf"\), \(\(:\)\(\ \)\), "\<\"The initialization of \
the method \\!\\(NDSolve`Splitting\\) failed. \\!\\(\\*ButtonBox[\\\"More\
\[Ellipsis]\\\", ButtonStyle->\\\"RefGuideLinkText\\\", ButtonFrame->None, \
ButtonData:>\\\"NDSolve::initf\\\"]\\)\"\>"}]], "Message",
  CellLabel->"From In[7]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 0.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 0.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[7]="]
}, Open  ]],

Cell[TextData[{
  "The equations can be extended by introducing a new \
\[OpenCurlyDoubleQuote]dummy\[CloseCurlyDoubleQuote] variable ",
  StyleBox["Z[T]", "Input"],
  " such that ",
  StyleBox["Z[T]==T", "Input"],
  " and specifying how it should be distributed in the split differential \
systems."
}], "Text"],

Cell[BoxData[{
    RowBox[{
      RowBox[{"Y1", " ", "=", " ", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_1\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(Y\_2[T]\)}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_2\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(Y\_2[T]\/4\)}], ",", 
            " ", \(\(Z'\)[T] \[Equal] 1\)}], "}"}]}], 
      ";"}], "\[IndentingNewLine]", 
    RowBox[{
      RowBox[{"Y2", " ", "=", " ", 
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_1\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], "\[Equal]", 
              "0"}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox[\(Y\_2\), "\[Prime]",
                  MultilineFunction->None], "[", "T", "]"}], 
              "\[Equal]", \(\(3\ Cos[Z[T]]\)\/10 + Y\_1[T] - Y\_1[T]\^3\)}], 
            ",", " ", \(\(Z'\)[T] \[Equal] 0\)}], "}"}]}], 
      ";"}], "\[IndentingNewLine]", \(eqs\  = \ 
      Join[system["\<System\>"[]], {\(Z'\)[T] \[Equal] 
            1}]; \), "\[IndentingNewLine]", \(ics\  = \ 
      Join[system["\<InitialConditions\>"[]], \ {Z[0] \[Equal] 
            1}]; \), "\[IndentingNewLine]", \(vars\  = \ 
      Join[system["\<DependentVariables\>"[]], \ {Z[
            T]}]; \), "\[IndentingNewLine]", \(time\  = \ 
      system["\<TimeData\>"[]]; \)}], "Input",
  CellLabel->"In[8]:="],

Cell[TextData[{
  "This defines a geometric splitting method that satisfies ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_1\  + \ \[Lambda]\_2\  = \
\(-\[Delta]\)\)]],
  " for any finite time interval, where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Lambda]\_2\)]],
  " are the Lyapunov exponents [",
  ButtonBox["M02",
    ButtonData:>{"References.nb", "M02"},
    ButtonStyle->"Hyperlink"],
  "]."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(splittingsol\  = \ 
      NDSolve[{eqs, \ ics}, \ vars, time, \ 
        StartingStepSize \[Rule] 1/10, \[IndentingNewLine]\ 
        Method \[Rule] {NDSolve`Splitting, \[IndentingNewLine]\ \
"\<Coefficients\>" \[Rule] 
              StrangSplittingCoefficients, \ "\<DifferenceOrder\>" \[Rule] 
              2, \[IndentingNewLine]"\<Equations\>" \[Rule] {Y2, Y1, 
                Y2}, \ "\<Method\>" \[Rule] {"\<LocallyExact\>"}}]\)], "Input",\

  CellLabel->"In[14]:="],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{"{", 
        RowBox[{
          RowBox[{\(Y\_1[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 10.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Y\_2[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 10.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}], ",", 
          RowBox[{\(Z[T]\), "\[Rule]", 
            RowBox[{
              TagBox[\(InterpolatingFunction[{{0.`, 10.`}}, "<>"]\),
                False,
                Editable->False], "[", "T", "]"}]}]}], "}"}], "}"}]], "Output",\

  CellLabel->"Out[14]="]
}, Open  ]],

Cell["Here is a plot of the solution.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(ParametricPlot[
      Evaluate[system["\<DependentVariables\>"[]]\  /. \ 
          First[splittingsol]], Evaluate[time]]\)], "Input",
  CellLabel->"In[15]:="],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.525814 0.140647 0.287939 0.0356104 [
[.10387 .27544 -6 -9 ]
[.10387 .27544 6 0 ]
[.24452 .27544 -6 -9 ]
[.24452 .27544 6 0 ]
[.38517 .27544 -6 -9 ]
[.38517 .27544 6 0 ]
[.66646 .27544 -3 -9 ]
[.66646 .27544 3 0 ]
[.80711 .27544 -3 -9 ]
[.80711 .27544 3 0 ]
[.94775 .27544 -3 -9 ]
[.94775 .27544 3 0 ]
[.51331 .02086 -24 -4.5 ]
[.51331 .02086 0 4.5 ]
[.51331 .10989 -12 -4.5 ]
[.51331 .10989 0 4.5 ]
[.51331 .19891 -24 -4.5 ]
[.51331 .19891 0 4.5 ]
[.51331 .37696 -18 -4.5 ]
[.51331 .37696 0 4.5 ]
[.51331 .46599 -6 -4.5 ]
[.51331 .46599 0 4.5 ]
[.51331 .55502 -18 -4.5 ]
[.51331 .55502 0 4.5 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.10387 .28794 m
.10387 .29419 L
s
[(-3)] .10387 .27544 0 1 Mshowa
.24452 .28794 m
.24452 .29419 L
s
[(-2)] .24452 .27544 0 1 Mshowa
.38517 .28794 m
.38517 .29419 L
s
[(-1)] .38517 .27544 0 1 Mshowa
.66646 .28794 m
.66646 .29419 L
s
[(1)] .66646 .27544 0 1 Mshowa
.80711 .28794 m
.80711 .29419 L
s
[(2)] .80711 .27544 0 1 Mshowa
.94775 .28794 m
.94775 .29419 L
s
[(3)] .94775 .27544 0 1 Mshowa
.125 Mabswid
.132 .28794 m
.132 .29169 L
s
.16013 .28794 m
.16013 .29169 L
s
.18826 .28794 m
.18826 .29169 L
s
.21639 .28794 m
.21639 .29169 L
s
.27265 .28794 m
.27265 .29169 L
s
.30078 .28794 m
.30078 .29169 L
s
.32891 .28794 m
.32891 .29169 L
s
.35704 .28794 m
.35704 .29169 L
s
.4133 .28794 m
.4133 .29169 L
s
.44143 .28794 m
.44143 .29169 L
s
.46956 .28794 m
.46956 .29169 L
s
.49768 .28794 m
.49768 .29169 L
s
.55394 .28794 m
.55394 .29169 L
s
.58207 .28794 m
.58207 .29169 L
s
.6102 .28794 m
.6102 .29169 L
s
.63833 .28794 m
.63833 .29169 L
s
.69459 .28794 m
.69459 .29169 L
s
.72272 .28794 m
.72272 .29169 L
s
.75085 .28794 m
.75085 .29169 L
s
.77898 .28794 m
.77898 .29169 L
s
.83524 .28794 m
.83524 .29169 L
s
.86337 .28794 m
.86337 .29169 L
s
.8915 .28794 m
.8915 .29169 L
s
.91963 .28794 m
.91963 .29169 L
s
.07574 .28794 m
.07574 .29169 L
s
.04761 .28794 m
.04761 .29169 L
s
.01949 .28794 m
.01949 .29169 L
s
.97588 .28794 m
.97588 .29169 L
s
.25 Mabswid
0 .28794 m
1 .28794 L
s
.52581 .02086 m
.53206 .02086 L
s
[(-7.5)] .51331 .02086 1 0 Mshowa
.52581 .10989 m
.53206 .10989 L
s
[(-5)] .51331 .10989 1 0 Mshowa
.52581 .19891 m
.53206 .19891 L
s
[(-2.5)] .51331 .19891 1 0 Mshowa
.52581 .37696 m
.53206 .37696 L
s
[(2.5)] .51331 .37696 1 0 Mshowa
.52581 .46599 m
.53206 .46599 L
s
[(5)] .51331 .46599 1 0 Mshowa
.52581 .55502 m
.53206 .55502 L
s
[(7.5)] .51331 .55502 1 0 Mshowa
.125 Mabswid
.52581 .03867 m
.52956 .03867 L
s
.52581 .05647 m
.52956 .05647 L
s
.52581 .07428 m
.52956 .07428 L
s
.52581 .09208 m
.52956 .09208 L
s
.52581 .12769 m
.52956 .12769 L
s
.52581 .1455 m
.52956 .1455 L
s
.52581 .1633 m
.52956 .1633 L
s
.52581 .18111 m
.52956 .18111 L
s
.52581 .21672 m
.52956 .21672 L
s
.52581 .23452 m
.52956 .23452 L
s
.52581 .25233 m
.52956 .25233 L
s
.52581 .27013 m
.52956 .27013 L
s
.52581 .30574 m
.52956 .30574 L
s
.52581 .32355 m
.52956 .32355 L
s
.52581 .34135 m
.52956 .34135 L
s
.52581 .35916 m
.52956 .35916 L
s
.52581 .39477 m
.52956 .39477 L
s
.52581 .41258 m
.52956 .41258 L
s
.52581 .43038 m
.52956 .43038 L
s
.52581 .44819 m
.52956 .44819 L
s
.52581 .4838 m
.52956 .4838 L
s
.52581 .5016 m
.52956 .5016 L
s
.52581 .51941 m
.52956 .51941 L
s
.52581 .53721 m
.52956 .53721 L
s
.52581 .00306 m
.52956 .00306 L
s
.52581 .57282 m
.52956 .57282 L
s
.52581 .59063 m
.52956 .59063 L
s
.52581 .60843 m
.52956 .60843 L
s
.25 Mabswid
.52581 0 m
.52581 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.5 Mabswid
.52581 .32355 m
.5889 .33193 L
.62783 .33735 L
.63928 .33848 L
.64569 .33927 L
.64887 .33961 L
.65183 .33983 L
.65761 .34015 L
.66393 .34058 L
.66671 .34083 L
.66932 .34098 L
.67206 .34106 L
.67352 .34108 L
.67509 .34109 L
.67791 .34108 L
.67946 .34107 L
.68092 .34107 L
.68375 .34109 L
.68631 .34116 L
.6878 .34119 L
.68937 .34118 L
.69219 .34109 L
.69376 .34101 L
.69523 .34091 L
.69855 .34064 L
.70403 .34017 L
.70669 .33999 L
.70914 .3398 L
.71196 .33947 L
.715 .33899 L
.72027 .33797 L
.72499 .33698 L
.72995 .33593 L
.73453 .33458 L
.73863 .33311 L
.74706 .32977 L
.75555 .32501 L
.76242 .32011 L
.76862 .31383 L
.77114 .31061 L
.77318 .30756 L
.77476 .30456 L
.77619 .30116 L
.77739 .29739 L
.77788 .29531 L
.77823 .29331 L
.77837 .29219 L
.77847 .29112 L
.77858 .28907 L
.7786 .28791 L
Mistroke
.77858 .28684 L
.77853 .28567 L
.77844 .28443 L
.7782 .28239 L
.77802 .28127 L
.77782 .28026 L
.77678 .27631 L
.77547 .27263 L
.77371 .26884 L
.76896 .26154 L
.76329 .25505 L
.75536 .24842 L
.74651 .24259 L
.73741 .23805 L
.72821 .23416 L
.71748 .23073 L
.70535 .22747 L
.69275 .22503 L
.68075 .22304 L
.67436 .22226 L
.66728 .22148 L
.65971 .22067 L
.65277 .22003 L
.63975 .21913 L
.62566 .2182 L
.59993 .21684 L
.57211 .2153 L
.54197 .21317 L
.50774 .20998 L
.47476 .20621 L
.44191 .20216 L
.4231 .19992 L
.40484 .19792 L
.3875 .19639 L
.37823 .19578 L
.37313 .19545 L
.36834 .19514 L
.36433 .19493 L
.35999 .19479 L
.35515 .19471 L
.3507 .1947 L
.34688 .19473 L
.34271 .19479 L
.33837 .19489 L
.33434 .19499 L
.32943 .19515 L
.32695 .19529 L
.32422 .19548 L
.31939 .19591 L
.31487 .19642 L
Mistroke
.30398 .19801 L
.29428 .19974 L
.28531 .20189 L
.27696 .20448 L
.26835 .20771 L
.25967 .21153 L
.24555 .21993 L
.23411 .22929 L
.22801 .23575 L
.22287 .24245 L
.21884 .24886 L
.21528 .25595 L
.21243 .26351 L
.21045 .27088 L
.20999 .27308 L
.20957 .27548 L
.20899 .27986 L
.20866 .28413 L
.20857 .28807 L
.2087 .29225 L
.20903 .29621 L
.20961 .30057 L
.21049 .30525 L
.21161 .30986 L
.21306 .31462 L
.21634 .32287 L
.21999 .32997 L
.22461 .33732 L
.23454 .34942 L
.24917 .36239 L
.26451 .37216 L
.2827 .38096 L
.29247 .38467 L
.3035 .388 L
.31402 .39087 L
.32428 .39333 L
.34393 .39669 L
.36783 .39992 L
.38032 .40111 L
.39402 .40203 L
.44233 .40493 L
.49645 .40813 L
.54886 .41237 L
.60194 .41778 L
.62596 .42026 L
.65287 .42289 L
.6658 .42401 L
.67972 .42494 L
.69299 .42563 L
.6989 .42587 L
Mistroke
.70517 .42608 L
.71234 .42623 L
.71895 .42626 L
.72604 .42618 L
.73007 .42607 L
.73383 .42592 L
.74064 .42558 L
.74689 .42517 L
.7541 .42456 L
.76094 .42383 L
.77481 .42182 L
.7824 .42034 L
.78918 .41877 L
.80247 .41509 L
.81444 .41092 L
.82782 .40498 L
.84129 .39713 L
.85335 .38848 L
.86352 .3795 L
.87973 .35921 L
.88694 .34674 L
.89252 .33414 L
.89457 .32823 L
.89641 .32175 L
.89771 .31601 L
.89883 .30978 L
.89972 .3031 L
.90027 .29681 L
.90054 .28931 L
.90044 .28239 L
.89998 .27535 L
.89911 .26778 L
.89791 .26059 L
.89647 .25415 L
.89435 .24674 L
.89157 .23889 L
.88538 .22512 L
.87749 .21136 L
.86745 .19777 L
.85666 .18643 L
.84539 .1767 L
.834 .16838 L
.82055 .1603 L
.79493 .14868 L
.77985 .14358 L
.76314 .13899 L
.74698 .13542 L
.73189 .13281 L
.71644 .13079 L
.69948 .12894 L
Mistroke
.68465 .12761 L
.67676 .12703 L
.6683 .12652 L
.65359 .12583 L
.63981 .12521 L
.6095 .12397 L
.47236 .11704 L
.43378 .11428 L
.39703 .11178 L
.37835 .11081 L
.36866 .1104 L
.35804 .11004 L
.35234 .10991 L
.34703 .10982 L
.34143 .10978 L
.33547 .10978 L
.32991 .10985 L
.32468 .10996 L
.31997 .11011 L
.31486 .11031 L
.30523 .11083 L
.29624 .1115 L
.28639 .11247 L
.27592 .11381 L
.26509 .11558 L
.25502 .11765 L
.23631 .12254 L
.21916 .12845 L
.20392 .13524 L
.18831 .14435 L
.17264 .15573 L
.16025 .16673 L
.14821 .18023 L
.13848 .19467 L
.13075 .2088 L
.12354 .22528 L
.12014 .23519 L
.11758 .24453 L
.11544 .25479 L
.11454 .2606 L
.11386 .26612 L
.11304 .27588 L
.11271 .28639 L
.11293 .2963 L
.11357 .30535 L
.11484 .3158 L
.11655 .32575 L
.12082 .34346 L
.12404 .35313 L
.12754 .3618 L
Mistroke
.13721 .38068 L
.14761 .3967 L
.16142 .41409 L
.17617 .42816 L
.21008 .45138 L
.22941 .46103 L
.25123 .46929 L
.26958 .47473 L
.28947 .4795 L
.31183 .48356 L
.33318 .48645 L
.35461 .48868 L
.37742 .49052 L
.39706 .49168 L
.41887 .49271 L
.51379 .49689 L
.60585 .50248 L
.62572 .5039 L
.6471 .50505 L
.67069 .50605 L
.6927 .50691 L
.70381 .50726 L
.71593 .50734 L
.72743 .50714 L
.73803 .50673 L
.74758 .5062 L
.75803 .50544 L
.77892 .50343 L
.79048 .50196 L
.80108 .50014 L
.8248 .49444 L
.84612 .48729 L
.86529 .47895 L
.88501 .46786 L
.90503 .4532 L
.92221 .43676 L
.93572 .41982 L
.94966 .39747 L
.95607 .38477 L
.96202 .37069 L
.96645 .35777 L
.97036 .34302 L
.97299 .32915 L
.97472 .31609 L
.97582 .30244 L
.9761 .29554 L
.97619 .28791 L
.97581 .27441 L
.97483 .26175 L
.97394 .25411 L
Mistroke
.9729 .24685 L
.97039 .23324 L
.96681 .21882 L
.9618 .2036 L
.95111 .1793 L
.93717 .15535 L
.92223 .1349 L
.9063 .11794 L
.88763 .10218 L
.866 .08762 L
.84663 .07718 L
.82403 .06734 L
.80421 .06036 L
.78244 .05422 L
.75928 .04927 L
.73725 .04548 L
.71104 .04176 L
.68284 .03869 L
.6587 .03679 L
.63188 .03488 L
.51402 .02691 L
.45832 .02247 L
.39842 .01808 L
.37344 .01654 L
.35967 .01575 L
.3466 .01512 L
.33913 .01489 L
.331 .01473 L
.31675 .01472 L
.30223 .01502 L
.28885 .01559 L
.27613 .01638 L
.26255 .01751 L
.24987 .01884 L
.23832 .02039 L
.22472 .02282 L
.20989 .02624 L
.18329 .03451 L
.15684 .04593 L
.13062 .0614 L
.10686 .07994 L
.08694 .10013 L
.0666 .12812 L
.05144 .15742 L
.03792 .19255 L
.0326 .21136 L
.03011 .22229 L
.0281 .23286 L
.02541 .25254 L
.02407 .271 L
Mistroke
.02381 .29081 L
.02417 .30191 L
.02482 .31211 L
.02656 .32892 L
.02939 .34724 L
.03332 .36609 L
.03786 .38351 L
.04302 .40012 L
.04944 .4174 L
.06387 .4465 L
.08484 .47749 L
.10634 .50201 L
.13058 .52392 L
.16006 .54402 L
.18999 .55938 L
.21832 .57017 L
.24478 .5778 L
.2704 .5839 L
.29911 .58933 L
.31525 .59172 L
.33022 .5935 L
.34577 .59491 L
.36279 .59634 L
.39399 .59886 L
.45311 .60279 L
.46385 .60332 L
Mfstroke
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  CellLabel->"From In[165]:=",
  ImageSize->{288, 177.938},
  ImageMargins->{{0, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40n007h00OU_oo0P00R?oo002F
ool00`00ooooo`27ool009Koo`03003ooooo08Ooo`00U_oo00<00?ooool0Qooo001=oolR000Dool0
0`00ooooo`03ool00`00ooooo`02ool30005ool00`00ooooo`27ool004;oo`/002;oo``000Soo`03
003ooooo00_oo`03003ooooo00;oo`03003ooooo08Ooo`00?_oo1000>Ooo40001_oo00<00?ooool0
0_oo0`00Qooo000jool4000nool40004ool00`00ooooo`04ool>0027ool003Koo`@004_oo`03003o
oooo00Koo`03003ooooo00Goo`03003ooooo00/007coo`00=?oo0P00C?oo00@00?ooool000Soo`03
003ooooo00Goo`03003ooooo00_oo``0073oo`00<_oo0P00C_oo10002?oo10001?oo00<00?ooool0
5ooo4000H?oo000`ool2001Tool2000Xool7001Iool002koo`8006Koo`03003ooooo02koo`P0057o
o`00;?oo0P00J?oo00<00?ooool0=_oo1`00B_oo000Zool2001Zool00`00ooooo`0mool70013ool0
02Soo`8006coo`03003ooooo04Coo`D003koo`009_oo0P00K_oo0P00B_oo0`00>ooo000Uool00`00
ooooo`1^ool00`00ooooo`1<ool3000hool002?oo`80077oo`03003ooooo04ooo`<003Goo`008_oo
00<00?ooool0LOoo00<00?ooool0D_oo0P00<ooo000Qool00`00ooooo`1bool00`00ooooo`1Dool2
000aool001ooo`8007Goo`8005Ooo`8002ooo`007_oo00<00?ooool0MOoo00<00?ooool0F?oo0P00
;Ooo000Mool00`00ooooo`1fool00`00ooooo`1Jool2000[ool001coo`03003ooooo07Ooo`03003o
oooo05coo`8002Woo`007?oo00<00?ooool0Mooo00<00?ooool0G_oo0P009ooo000Kool00`00oooo
o`1hool2001Qool2000Uool001[oo`03003ooooo07Woo`03003ooooo06;oo`03003ooooo02;oo`00
6Ooo00<00?ooool0N_oo00<00?ooool0Hooo0P008_oo000Hool00`00ooooo`1cool30005ool00`00
ooooo`1Uool00`00ooooo`0Oool001Soo`03003ooooo07Koo`03003ooooo00;oo`03003ooooo06Ko
o`03003ooooo01koo`005ooo00<00?ooool0>_oo6`008_oo00<00?ooool00_oo0`00Iooo0P007_oo
000Fool00`00ooooo`0eool6000KoolC0006ool40002ool30005ool00`00ooooo`1Yool00`00oooo
o`0Kool001Goo`03003ooooo033oo`H003Coo`l000Goo`03003ooooo06[oo`03003ooooo01[oo`00
5Ooo00<00?ooool0;_oo0P00A_oo00<00?ooool05000Gooo00<00?ooool06Ooo000Dool00`00oooo
o`0[ool40018ool40004ool00`00ooooo`0<oolD001<ool00`00ooooo`0Hool001?oo`03003ooooo
02Soo`@005Coo`80027ooa8003_oo`03003ooooo01Ooo`004ooo00<00?ooool09_oo0P00F?oo00<0
0?ooool0<_oo1000>?oo00<00?ooool05_oo000Bool00`00ooooo`0Uool2001Jool00`00ooooo`0f
ool9000`ool00`00ooooo`0Eool001;oo`03003ooooo02?oo`8005coo`03003ooooo03ooo`@002go
o`03003ooooo01Coo`004Ooo00<00?ooool08ooo00<00?ooool0G?oo00<00?ooool0@ooo0P00:ooo
00<00?ooool05?oo000Aool00`00ooooo`0Rool00`00ooooo`1Mool20016ool3000Yool00`00oooo
o`0Cool0013oo`03003ooooo027oo`80063oo`03003ooooo04Soo`@002Koo`03003ooooo01;oo`00
4?oo00<00?ooool07ooo0P00H_oo00<00?ooool0C?oo0P009?oo00<00?ooool04_oo000?ool00`00
ooooo`0Oool00`00ooooo`1Rool00`00ooooo`1>ool3000Rool00`00ooooo`0Aool000ooo`03003o
oooo01koo`03003ooooo06?oo`03003ooooo057oo`80027oo`03003ooooo013oo`003ooo00<00?oo
ool07Ooo00<00?ooool0I?oo0P00E?oo00<00?ooool07_oo00<00?ooool04?oo000>ool00`00oooo
o`0Mool00`00ooooo`1Uool00`00ooooo`1Dool2000Oool00`00ooooo`0?ool000koo`03003ooooo
01coo`03003ooooo06Koo`03003ooooo05Koo`03003ooooo01coo`03003ooooo00ooo`003Ooo00<0
0?ooool07?oo00<00?ooool0Iooo00<00?ooool0Eooo0P007Ooo00<00?ooool03_oo000=ool00`00
ooooo`0Kool00`00ooooo`1Xool00`00ooooo`1Iool00`00ooooo`0Jool00`00ooooo`0>ool000go
o`03003ooooo01[oo`03003ooooo06Woo`8005_oo`03003ooooo01[oo`03003ooooo00goo`003?oo
00<00?ooool06ooo00<00?ooool0JOoo00<00?ooool0Fooo00<00?ooool06Ooo00<00?ooool03Ooo
000<ool00`00ooooo`0Jool00`00ooooo`1Zool00`00ooooo`1Lool00`00ooooo`0Iool00`00oooo
o`0<ool000_oo`03003ooooo01_oo`03003ooooo06[oo`03003ooooo05goo`03003ooooo01Soo`03
003ooooo00coo`002ooo00<00?ooool06_oo00<00?ooool0<_oo3@006?oo10000ooo00<00?ooool0
0_oo0`001Ooo00<00?ooool0G_oo00<00?ooool06?oo00<00?ooool02ooo000;ool00`00ooooo`0I
ool00`00ooooo`0/ool8000<ool=000;ool00`00ooooo`0<ool00`00ooooo`02ool00`00ooooo`1O
ool00`00ooooo`0Gool00`00ooooo`0;ool000[oo`03003ooooo01[oo`03003ooooo02Ooo`D0027o
o`X000;oo`03003ooooo00_oo`03003ooooo00;oo`<0063oo`03003ooooo01Ooo`03003ooooo00[o
o`002_oo00<00?ooool06Ooo00<00?ooool09Ooo0`00:ooo100000?oo`0000000`002?oo0`001Ooo
00<00?ooool0H?oo00<00?ooool05ooo00<00?ooool02_oo0009ool00`00ooooo`0Iool00`00oooo
o`0Tool2000gool800000ooo003oo`06ool00`00ooooo`1Qool00`00ooooo`0Fool00`00ooooo`0:
ool000Woo`03003ooooo01Woo`03003ooooo02;oo`8003Koo`04003ooooo0007ool:001Tool00`00
ooooo`0Fool00`00ooooo`09ool000Woo`03003ooooo01Soo`03003ooooo027oo`8003Woo`8000Wo
o`@000Cooa8005?oo`03003ooooo01Koo`03003ooooo00Woo`002?oo00<00?ooool06Ooo00<00?oo
ool08?oo00<00?ooool0C?oo0P004?oo4@00@ooo00<00?ooool05_oo00<00?ooool02?oo0008ool0
0`00ooooo`0Hool00`00ooooo`0Pool00`00ooooo`1=ool00`00ooooo`0Pool<000gool00`00oooo
o`0Fool00`00ooooo`08ool000Soo`03003ooooo01Soo`03003ooooo01koo`80053oo`03003ooooo
02coo`<003Goo`03003ooooo01Goo`03003ooooo00Soo`002?oo00<00?ooool06?oo00<00?ooool0
7Ooo00<00?ooool0D?oo00<00?ooool0;ooo1000<Ooo00<00?ooool05_oo00<00?ooool01ooo0008
ool00`00ooooo`0Gool00`00ooooo`0Mool00`00ooooo`1Aool00`00ooooo`0cool3000_ool00`00
ooooo`0Eool00`00ooooo`07ool000Soo`03003ooooo01Ooo`03003ooooo01coo`03003ooooo05;o
o`8003Ooo`<002coo`03003ooooo01Goo`03003ooooo00Ooo`002?oo00<00?ooool05ooo00<00?oo
ool07?oo00<00?ooool0D_oo00<00?ooool0>Ooo0`00:_oo00<00?ooool05?oo00<00?ooool01ooo
0008ool00`00ooooo`0Gool00`00ooooo`0Kool00`00ooooo`1Cool00`00ooooo`0lool2000Xool0
0`00ooooo`0Dool00`00ooooo`07ool000Ooo`03003ooooo01Soo`03003ooooo01_oo`03003ooooo
05?oo`03003ooooo03koo`03003ooooo02Koo`03003ooooo01?oo`03003ooooo00Ooo`001ooo00<0
0?ooool05_oo0`007?oo00<00?ooool01Ooo10009?oo10008ooo00<00?ooool09?oo10005ooo00<0
0?ooool02_oo10005ooo00<00?ooool02ooo0P001_oo00<00?ooool01ooo0007ool00`00ooooo`0E
ool01000ool000006ooo00<00?ooool01_oo00<00?ooool09ooo00<00?ooool08_oo0P009ooo00<0
0?ooool05ooo00<00?ooool02Ooo00<00?ooool06Ooo00<00?ooool02Ooo00@00?ooool000Koo`03
003ooooo00Koo`001ooo00<00?ooool05ooo0P006_oo00<00?ooool02?oo00<00?ooool09_oo00<0
0?ooool08_oo00<00?ooool09_oo00<00?ooool06?oo00<00?ooool02Ooo00<00?ooool06?oo00<0
0?ooool03?oo00<00?ooool01?oo00<00?ooool01_oo0007ool00`00ooooo`0?ool40003ool2000K
ool01000oooooooo10001?oo00<00?ooool07Ooo10001?oo00<00?ooool08_oo00<00?ooool09_oo
00<00?ooool06Ooo0P002_oo00<00?ooool05ooo00<00?ooool02_oo0P001ooo00<00?ooool01_oo
0007ool00`00ooooo`0Gool2000Jool00`00ooooo`0:ool00`00ooooo`0Tool00`00ooooo`0Rool0
0`00ooooo`0Vool00`00ooooo`0Jool00`00ooooo`09ool00`00ooooo`0Fool00`00ooooo`0<ool0
0`00ooooo`04ool00`00ooooo`06ool000Ooo`03003ooooo01Goo`800003ool00?oo01Woo`03003o
oooo00Ooo`04003ooooo000Tool3000Tool00`00ooooo`0Tool3000Mool00`00ooooo`05ool01000
ooooo`006?oo00<00?ooool02Ooo00@00?ooool000Koo`03003ooooo00Koo`001ooo00<00?ooool0
5_oo0P006_oo00<00?ooool02Ooo0P009ooo00<00?ooool08_oo0P009ooo00<00?ooool06ooo00<0
0?ooool01_oo0P006Ooo00<00?ooool02_oo0P001ooo00<00?ooool01_oo0007ool00`00ooooo`0F
ool00`00ooooo`0Iool00`00ooooo`1Gool00`00ooooo`15ool00`00ooooo`0Qool00`00ooooo`0B
ool00`00ooooo`06ool000Ooo`03003ooooo01Koo`03003ooooo01Woo`03003ooooo05Ooo`03003o
oooo04Goo`03003ooooo027oo`03003ooooo01;oo`03003ooooo00Koo`001ooo00<00?ooool05_oo
00<00?ooool06Ooo00<00?ooool0Eooo00<00?ooool0A_oo00<00?ooool08?oo00<00?ooool04_oo
00<00?ooool01_oo0007ool00`00ooooo`0Fool00`00ooooo`0Iool00`00ooooo`1Gool00`00oooo
o`16ool00`00ooooo`0Pool00`00ooooo`0Bool00`00ooooo`06ool00?l00200007oo`001Ooo00<0
0?oo00001Ooo00<00?ooool01_oo00<00?ooool01Ooo00<00?oo00001Ooo00<00?ooool01Ooo00<0
0?ooool01Ooo00<00?ooool00ooo00<00?oo00001ooo00<00?ooool01Ooo00<00?ooool01Ooo00<0
0?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<0
0?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<0
0?ooool01_oo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<0
0?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<00?ooool01Ooo00<0
0?ooool01Ooo00@00?ooool000Coo`03003ooooo00Goo`03003ooooo00Goo`03003ooooo00Koo`00
1ooo00<00?ooool05?oo00<00?oo00006ooo00<00?ooool01ooo00<00?ooool09Ooo00<00?ooool0
9Ooo00<00?ooool09_oo00<00?ooool07Ooo00<00?ooool01Ooo00<00?ooool06?oo00<00?ooool0
2_oo00<00?ooool01Ooo00<00?ooool01_oo0007ool00`00ooooo`0Fool00`00ooooo`0Iool00`00
ooooo`1Gool00`00ooooo`15ool00`00ooooo`0Qool00`00ooooo`0Bool00`00ooooo`06ool000Oo
o`03003ooooo01Koo`03003ooooo01Woo`03003ooooo05Ooo`03003ooooo04Goo`03003ooooo023o
o`03003ooooo01?oo`03003ooooo00Koo`001ooo00<00?ooool05_oo00<00?ooool06Ooo00<00?oo
ool0Eooo0P00A_oo00<00?ooool08?oo00<00?ooool04ooo00<00?ooool01_oo0007ool00`00oooo
o`0Fool00`00ooooo`0Iool00`00ooooo`1Gool00`00ooooo`14ool00`00ooooo`0Qool00`00oooo
o`0Cool00`00ooooo`06ool000Ooo`03003ooooo01Ooo`03003ooooo01Soo`03003ooooo05Ooo`03
003ooooo04?oo`03003ooooo02;oo`03003ooooo01?oo`03003ooooo00Koo`001ooo00<00?ooool0
5ooo00<00?ooool06Ooo00<00?ooool0E_oo00<00?ooool0@_oo00<00?ooool08ooo00<00?ooool0
4ooo00<00?ooool01_oo0007ool00`00ooooo`0Gool00`00ooooo`0Iool00`00ooooo`1Fool00`00
ooooo`11ool00`00ooooo`0Tool00`00ooooo`0Cool00`00ooooo`06ool000Ooo`03003ooooo01Oo
o`03003ooooo01[oo`03003ooooo05Goo`<0043oo`03003ooooo02Coo`03003ooooo01Coo`03003o
oooo00Koo`001ooo00<00?ooool05ooo00<00?ooool06_oo00<00?ooool0EOoo00<00?ooool01P00
>?oo0P009ooo00<00?ooool04ooo00<00?ooool01ooo0007ool00`00ooooo`0Gool00`00ooooo`0K
ool00`00ooooo`1Dool00`00ooooo`06ool6000`ool2000Yool00`00ooooo`0Cool00`00ooooo`07
ool000Ooo`03003ooooo01Ooo`03003ooooo01_oo`03003ooooo05Coo`03003ooooo00coo`T002Co
o`<002[oo`03003ooooo01Coo`03003ooooo00Ooo`001ooo00<00?ooool06?oo00<00?ooool06ooo
00<00?ooool0Dooo00<00?ooool05Ooo2P005_oo1000;Ooo00<00?ooool05?oo00<00?ooool01ooo
0008ool00`00ooooo`0Gool00`00ooooo`0Lool00`00ooooo`1Bool2000PoolF000aool00`00oooo
o`0Dool00`00ooooo`07ool000Soo`03003ooooo01Ooo`03003ooooo01coo`03003ooooo05;oo`03
003ooooo06Goo`03003ooooo01Goo`03003ooooo00Ooo`002?oo00<00?ooool05ooo00<00?ooool0
7Ooo00<00?ooool0DOoo00<00?ooool0IOoo00<00?ooool05Ooo00<00?ooool01ooo0008ool00`00
ooooo`0Hool00`00ooooo`0Mool00`00ooooo`1@ool00`00ooooo`1Tool00`00ooooo`0Fool00`00
ooooo`07ool000Soo`03003ooooo01Soo`03003ooooo01koo`03003ooooo04ooo`03003ooooo06Co
o`03003ooooo01Goo`03003ooooo00Soo`002Ooo00<00?ooool05ooo00<00?ooool07ooo0P00Cooo
0P00I?oo00<00?ooool05_oo00<00?ooool02?oo0009ool00`00ooooo`0Hool00`00ooooo`0Pool0
0`00ooooo`1<ool00`00ooooo`1Rool00`00ooooo`0Gool00`00ooooo`08ool000Woo`03003ooooo
01Soo`03003ooooo027oo`8004coo`03003ooooo067oo`03003ooooo01Ooo`03003ooooo00Woo`00
2Ooo00<00?ooool06Ooo00<00?ooool08_oo0P00=_oo10000ooo00<00?ooool00_oo0`001Ooo00<0
0?ooool0HOoo00<00?ooool05ooo00<00?ooool02Ooo000:ool00`00ooooo`0Hool00`00ooooo`0T
ool00`00ooooo`0cool00`00ooooo`0<ool00`00ooooo`02ool00`00ooooo`1Pool00`00ooooo`0G
ool00`00ooooo`0:ool000[oo`03003ooooo01Woo`03003ooooo02Coo`8003Coo`03003ooooo00_o
o`03003ooooo00;oo`<005ooo`03003ooooo01Soo`03003ooooo00[oo`002ooo00<00?ooool06?oo
00<00?ooool09_oo0P00<ooo00<00?ooool01ooo0`001Ooo00<00?ooool0G_oo00<00?ooool06Ooo
00<00?ooool02_oo000;ool00`00ooooo`0Iool00`00ooooo`0Wool2000bool00`00ooooo`06ool0
0`00ooooo`05ool00`00ooooo`1Mool00`00ooooo`0Iool00`00ooooo`0;ool000_oo`03003ooooo
01[oo`03003ooooo02Soo`<002coo`04003ooooo0008ool00`00ooooo`05ool00`00ooooo`1Lool0
0`00ooooo`0Jool00`00ooooo`0;ool000_oo`03003ooooo01[oo`03003ooooo02_oo`H002Ooo`80
00Woo`@000Coo`03003ooooo05_oo`03003ooooo01[oo`03003ooooo00coo`003?oo00<00?ooool0
6_oo00<00?ooool0<?oo1000=_oo00<00?ooool0FOoo0P007Ooo00<00?ooool03?oo000<ool00`00
ooooo`0Kool00`00ooooo`0cool6000`ool2001Iool00`00ooooo`0Lool00`00ooooo`0=ool000co
o`03003ooooo01coo`03003ooooo03Soo`T002Ooo`03003ooooo05Koo`8001koo`03003ooooo00ko
o`003?oo00<00?ooool07Ooo00<00?ooool0@?oo5`004?oo00<00?ooool0EOoo00<00?ooool07_oo
00<00?ooool03_oo000=ool00`00ooooo`0Lool00`00ooooo`1Gool<0004ool00`00ooooo`1Bool3
000Pool00`00ooooo`0?ool000goo`03003ooooo01goo`03003ooooo06;oo`P004koo`<002?oo`03
003ooooo00ooo`003_oo00<00?ooool07Ooo00<00?ooool0IOoo0P000_oo2`00?ooo10009Ooo00<0
0?ooool04?oo000>ool00`00ooooo`0Nool00`00ooooo`1Tool00`00ooooo`0<ool;000aool3000Y
ool00`00ooooo`0@ool000koo`03003ooooo01ooo`03003ooooo06?oo`03003ooooo01Ooo`L002Go
o`D002_oo`03003ooooo017oo`003ooo00<00?ooool07ooo00<00?ooool0H_oo00<00?ooool07_oo
4P002Ooo2P00;ooo00<00?ooool04_oo000?ool00`00ooooo`0Pool00`00ooooo`1Qool00`00oooo
o`0`ool9000hool00`00ooooo`0Cool0013oo`03003ooooo023oo`80067oo`8007;oo`03003ooooo
01?oo`004?oo00<00?ooool08_oo00<00?ooool0G_oo00<00?ooool0L?oo00<00?ooool05?oo000A
ool00`00ooooo`0Rool00`00ooooo`1Mool00`00ooooo`1_ool00`00ooooo`0Eool0017oo`03003o
oooo02?oo`8005goo`03003ooooo06koo`03003ooooo01Koo`004_oo00<00?ooool09?oo00<00?oo
ool0F_oo00<00?ooool0K?oo0P006Ooo000Cool00`00ooooo`0Tool2001Jool2001/ool00`00oooo
o`0Iool001?oo`03003ooooo02Koo`03003ooooo05Ooo`03003ooooo06[oo`03003ooooo01[oo`00
5?oo00<00?ooool09_oo0P00Eooo00<00?ooool0JOoo00<00?ooool06ooo000Eool00`00ooooo`0W
ool2001=ool30005ool00`00ooooo`1Wool2000Nool001Goo`03003ooooo02Woo`<004goo`03003o
oooo00;oo`03003ooooo06Koo`03003ooooo01koo`005_oo00<00?ooool0:ooo0`00B_oo00<00?oo
ool00_oo0`00IOoo00<00?ooool07ooo000Gool00`00ooooo`0]ool40013ool30005ool00`00oooo
o`1Tool00`00ooooo`0Pool001Ooo`03003ooooo037oo`@003ooo`03003ooooo00Goo`03003ooooo
06;oo`8002?oo`006?oo00<00?ooool0=?oo0`00??oo00<00?ooool01Ooo00<00?ooool0H?oo0P00
9Ooo000Iool00`00ooooo`0fool4000hool40004ool00`00ooooo`1Nool2000Wool001[oo`03003o
oooo03Woo`H003[oo`8005coo`<002Woo`006ooo00<00?ooool0?_oo3@00;Ooo00<00?ooool0F?oo
0`00;?oo000Kool00`00ooooo`1;ool<000Qool00`00ooooo`1Eool3000_ool001coo`03003ooooo
05Kooa40013oo`03003ooooo05;oo`<003;oo`007Ooo00<00?ooool0I_oo6P00Aooo1000=Ooo000N
ool00`00ooooo`1eool20008ool@000aool6000iool001ooo`03003ooooo07Coo`03003ooooo01Oo
oa<001;oo``003ooo`008?oo00<00?ooool0Looo00<00?ooool0:_oo4P00Booo000Qool00`00oooo
o`1bool00`00ooooo`27ool002;oo`03003ooooo077oo`03003ooooo08Ooo`008ooo00<00?ooool0
L?oo0P00R?oo000Tool00`00ooooo`1_ool00`00ooooo`27ool002Goo`03003ooooo06koo`03003o
oooo08Ooo`009_oo0P00K_oo00<00?ooool0Qooo000Xool2001/ool00`00ooooo`27ool002[oo`80
06[oo`8008Soo`00;?oo0P00J?oo00<00?ooool0Qooo000^ool00`00ooooo`1Uool00`00ooooo`27
ool002ooo`8005;oo`03003ooooo00?oo`03003ooooo00;oo`<000Goo`03003ooooo08Ooo`00<Ooo
00<00?ooool0Cooo00<00?ooool02ooo00<00?ooool00_oo00<00?ooool0Qooo000bool2001@ool0
0`00ooooo`0:ool00`00ooooo`02ool30027ool003Coo`8004koo`03003ooooo00Ooo`<000Goo`03
003ooooo08Ooo`00=_oo0P00COoo00<00?ooool01_oo00<00?ooool01Ooo00<00?ooool0Qooo000h
ool20018ool01000ooooo`002?oo00<00?ooool01Ooo00<00?ooool0Qooo000jool30015ool40008
ool40004ool00`00ooooo`27ool003goo`<005Koo`03003ooooo08Ooo`00@?oo1000D_oo0P00R?oo
0014ool4001>ool00`00ooooo`27ool004Soo`@004[oo`03003ooooo08Ooo`00C?oo1@00AOoo00<0
0?ooool0Qooo001Aool7000nool00`00ooooo`27ool005Soo`T003Goo`8008Soo`00HOoo3000:Ooo
00<00?ooool0Qooo001]ool=000Lool00`00ooooo`27ool007[oo``0013oo`03003ooooo08Ooo`00
U_oo00<00?ooool0Qooo002Fool20028ool009Koo`03003ooooo08Ooo`00U_oo00<00?ooool0Qooo
0000\
\>"],
  ImageRangeCache->{{{93.5625, 452.312}, {521.625, 300.438}} -> {-5.60687, \
15.4921, 0.024835, 0.0980883}}]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Option summary", "Section",
  CellTags->"s:3"],

Cell[TextData[{
  "The default coefficient choice in ",
  StyleBox["Composition", "MR"],
  " tries to automatically select between ",
  StyleBox["SymmetricCompositionCoefficients", "MR"],
  " and ",
  StyleBox["SymmetricCompositionSymmetricMethodCoefficients", "MR"],
  " depending on the properties of the methods specified using the ",
  StyleBox["Method", "MR"],
  " option."
}], "Text"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {
            StyleBox["Coefficients",
              "MR"], 
            StyleBox["Automatic",
              
              "MR"], \(specifies\ the\ coefficients\ to\ use\ in\ the\ \
composition\ method\)},
          {
            StyleBox["DifferenceOrder",
              "MR"], 
            StyleBox["Automatic",
              
              "MR"], \(specifies\ the\ order\ of\ local\ accuracy\ of\ the\ \
method\)},
          {
            StyleBox["Method",
              "MR"], 
            StyleBox["None",
              
              "MR"], \(specifies\ the\ base\ methods\ to\ use\ in\ the\ \
numerical\ integration\)}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  "Options of the method ",
  StyleBox["Composition", "MR"],
  ". "
}], "Caption"],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[StyleBox["option name", "SO10"]]], Cell[TextData[
            StyleBox["default value", "SO10"]]], " "},
          {
            StyleBox["Coefficients",
              "MR"], 
            StyleBox[\({}\),
              
              "MR"], \(specifies\ the\ coefficients\ to\ use\ in\ the\ \
splitting\ method\)},
          {
            StyleBox["DifferenceOrder",
              "MR"], 
            StyleBox["Automatic",
              
              "MR"], \(specifies\ the\ order\ of\ local\ accuracy\ of\ the\ \
method\)},
          {
            StyleBox["Equations",
              "MR"], 
            StyleBox[\({}\),
              "MR"], Cell[
            "specifies the way in which the equations should be split", 
              "Text"]},
          {
            StyleBox["Method",
              "MR"], 
            "None", \(specifies\ the\ base\ methods\ to\ use\ in\ the\ \
numerical\ integration\)}
          }], TextForm]], "DefinitionBox3Col",
  GridBoxOptions->{RowLines->{0.5, False}}],

Cell[TextData[{
  "Options of the method ",
  StyleBox["Splitting", "MR"],
  "."
}], "Caption"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.0 for X",
ScreenRectangle->{{0, 1600}, {0, 1200}},
AutoGeneratedPackage->None,
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{650, 500},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Composition and Splitting",
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
Magnification->1,
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "c:1"->{
    Cell[1776, 53, 63, 1, 87, "Chapter",
      CellTags->"c:1"]},
  "s:1"->{
    Cell[1864, 58, 50, 1, 67, "Section",
      CellTags->"s:1"]},
  "s:2"->{
    Cell[22690, 648, 46, 1, 70, "Section",
      CellTags->"s:2"]},
  "s:3"->{
    Cell[77241, 2452, 52, 1, 67, "Section",
      CellTags->"s:3"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"c:1", 80680, 2565},
  {"s:1", 80757, 2568},
  {"s:2", 80834, 2571},
  {"s:3", 80913, 2574}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 63, 1, 87, "Chapter",
  CellTags->"c:1"],

Cell[CellGroupData[{
Cell[1864, 58, 50, 1, 67, "Section",
  CellTags->"s:1"],
Cell[1917, 61, 282, 5, 56, "Text"],

Cell[CellGroupData[{
Cell[2224, 70, 33, 0, 41, "Subsection"],
Cell[2260, 72, 285, 9, 24, "Text"],

Cell[CellGroupData[{
Cell[2570, 85, 36, 0, 34, "Subsubsection"],
Cell[2609, 87, 113, 3, 24, "Text"],
Cell[2725, 92, 193, 4, 40, "Text"],
Cell[2921, 98, 389, 12, 28, "Text"],
Cell[3313, 112, 212, 8, 25, "Text"],
Cell[3528, 122, 282, 5, 29, "Text"],
Cell[3813, 129, 328, 8, 25, "Text"],
Cell[4144, 139, 293, 9, 24, "Text"],
Cell[4440, 150, 57, 0, 24, "Text"],
Cell[4500, 152, 82, 1, 26, "BulletedList"],
Cell[4585, 155, 259, 8, 26, "BulletedList"],
Cell[4847, 165, 59, 0, 26, "BulletedList"]
}, Open  ]],

Cell[CellGroupData[{
Cell[4943, 170, 34, 0, 34, "Subsubsection"],
Cell[4980, 172, 259, 9, 40, "Text"],
Cell[5242, 183, 202, 7, 24, "Text"],
Cell[5447, 192, 260, 9, 40, "Text"],
Cell[5710, 203, 142, 5, 24, "Text"],
Cell[5855, 210, 286, 5, 29, "Text"],
Cell[6144, 217, 139, 5, 24, "Text"],
Cell[6286, 224, 192, 4, 40, "Text"],
Cell[6481, 230, 292, 8, 42, "Text"],
Cell[6776, 240, 99, 3, 70, "Text"],
Cell[6878, 245, 589, 17, 70, "Text"],
Cell[7470, 264, 107, 3, 70, "Text"],
Cell[7580, 269, 204, 5, 70, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[7833, 280, 36, 0, 70, "Subsection"],
Cell[7872, 282, 175, 5, 70, "Text"],
Cell[8050, 289, 83, 1, 70, "BulletedList"],
Cell[8136, 292, 146, 3, 70, "BulletedList"],
Cell[8285, 297, 189, 5, 70, "BulletedList"],
Cell[8477, 304, 170, 4, 70, "BulletedList"],
Cell[8650, 310, 157, 3, 70, "Text"],
Cell[8810, 315, 486, 17, 70, "Text"],
Cell[9299, 334, 852, 28, 70, "Text"],

Cell[CellGroupData[{
Cell[10176, 366, 39, 0, 70, "Subsubsection"],
Cell[10218, 368, 134, 3, 70, "Text"],
Cell[10355, 373, 7684, 157, 70, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[18076, 535, 39, 0, 70, "Subsubsection"],
Cell[18118, 537, 206, 5, 70, "Text"],
Cell[18327, 544, 4302, 97, 70, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[22690, 648, 46, 1, 70, "Section",
  CellTags->"s:2"],
Cell[22739, 651, 420, 12, 70, "Text"],
Cell[23162, 665, 157, 4, 70, "Input",
  InitializationCell->True],
Cell[23322, 671, 168, 5, 70, "Text"],
Cell[23493, 678, 235, 5, 70, "Input",
  InitializationCell->True],
Cell[23731, 685, 65, 0, 70, "Text"],
Cell[23799, 687, 138, 3, 70, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[23962, 694, 42, 0, 70, "Subsection"],

Cell[CellGroupData[{
Cell[24029, 698, 44, 0, 70, "Subsubsection"],
Cell[24076, 700, 270, 6, 70, "Text"],
Cell[24349, 708, 267, 6, 70, "Text"],
Cell[24619, 716, 423, 17, 40, "Text"],

Cell[CellGroupData[{
Cell[25067, 737, 114, 2, 26, "Input"],
Cell[25184, 741, 821, 20, 67, "Output"]
}, Open  ]],
Cell[26020, 764, 203, 4, 40, "Text"],
Cell[26226, 770, 310, 6, 90, "Input"],
Cell[26539, 778, 153, 3, 40, "Text"],

Cell[CellGroupData[{
Cell[26717, 785, 695, 13, 122, "Input"],
Cell[27415, 800, 544, 15, 41, "Output"]
}, Open  ]],
Cell[27974, 818, 198, 5, 40, "Text"],
Cell[28175, 825, 70, 0, 70, "Text"],

Cell[CellGroupData[{
Cell[28270, 829, 105, 2, 70, "Input"],
Cell[28378, 833, 151, 4, 70, "Output"]
}, Open  ]],
Cell[28544, 840, 123, 3, 70, "Text"],

Cell[CellGroupData[{
Cell[28692, 847, 181, 4, 70, "Input"],
Cell[28876, 853, 544, 15, 70, "Output"]
}, Open  ]],
Cell[29435, 871, 123, 3, 70, "Text"],

Cell[CellGroupData[{
Cell[29583, 878, 100, 2, 70, "Input"],
Cell[29686, 882, 151, 4, 70, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[29886, 892, 59, 0, 70, "Subsubsection"],
Cell[29948, 894, 303, 8, 70, "Text"],

Cell[CellGroupData[{
Cell[30276, 906, 785, 17, 70, "Input"],
Cell[31064, 925, 544, 15, 70, "Output"]
}, Open  ]],
Cell[31623, 943, 70, 0, 70, "Text"],

Cell[CellGroupData[{
Cell[31718, 947, 105, 2, 70, "Input"],
Cell[31826, 951, 151, 4, 70, "Output"]
}, Open  ]],
Cell[31992, 958, 147, 3, 70, "Text"],

Cell[CellGroupData[{
Cell[32164, 965, 872, 17, 70, "Input"],
Cell[33039, 984, 544, 15, 70, "Output"]
}, Open  ]],
Cell[33598, 1002, 125, 3, 70, "Text"],

Cell[CellGroupData[{
Cell[33748, 1009, 100, 2, 70, "Input"],
Cell[33851, 1013, 151, 4, 70, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[34063, 1024, 36, 0, 70, "Subsection"],
Cell[34102, 1026, 222, 4, 70, "Text"],
Cell[34327, 1032, 174, 5, 70, "Text"],
Cell[34504, 1039, 196, 6, 24, "Text"],

Cell[CellGroupData[{
Cell[34725, 1049, 57, 0, 34, "Subsubsection"],
Cell[34785, 1051, 214, 4, 40, "Text"],
Cell[35002, 1057, 526, 11, 138, "Input"],
Cell[35531, 1070, 460, 10, 90, "Input"],

Cell[CellGroupData[{
Cell[36016, 1084, 97, 2, 26, "Input"],
Cell[36116, 1088, 152, 4, 57, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[36305, 1097, 559, 11, 122, "Input"],
Cell[36867, 1110, 152, 4, 57, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[37068, 1120, 77, 0, 34, "Subsubsection"],
Cell[37148, 1122, 146, 3, 40, "Text"],

Cell[CellGroupData[{
Cell[37319, 1129, 126, 3, 26, "Input"],
Cell[37448, 1134, 1114, 25, 129, "Output"]
}, Open  ]],
Cell[38577, 1162, 220, 4, 40, "Text"],
Cell[38800, 1168, 123, 3, 24, "Text"],
Cell[38926, 1173, 315, 6, 90, "Input"],

Cell[CellGroupData[{
Cell[39266, 1183, 60, 2, 26, "Input"],
Cell[39329, 1187, 692, 19, 78, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[40058, 1211, 60, 2, 26, "Input"],
Cell[40121, 1215, 641, 18, 25, "Output"]
}, Open  ]],
Cell[40777, 1236, 207, 8, 24, "Text"],
Cell[40987, 1246, 203, 5, 40, "Text"],
Cell[41193, 1253, 176, 5, 40, "Text"],
Cell[41372, 1260, 505, 9, 90, "Input"],
Cell[41880, 1271, 219, 7, 24, "Text"],

Cell[CellGroupData[{
Cell[42124, 1282, 527, 9, 106, "Input"],
Cell[42654, 1293, 764, 20, 57, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[43467, 1319, 49, 0, 34, "Subsubsection"],
Cell[43519, 1321, 254, 6, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[43810, 1332, 42, 0, 34, "Subsubsection"],
Cell[43855, 1334, 216, 6, 24, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[44108, 1345, 53, 0, 34, "Subsubsection"],
Cell[44164, 1347, 110, 3, 24, "Text"],

Cell[CellGroupData[{
Cell[44299, 1354, 112, 2, 26, "Input"],
Cell[44414, 1358, 842, 20, 68, "Output"]
}, Open  ]],
Cell[45271, 1381, 55, 0, 24, "Text"],
Cell[45329, 1383, 1147, 31, 80, "Input"],
Cell[46479, 1416, 138, 3, 40, "Text"],

Cell[CellGroupData[{
Cell[46642, 1423, 439, 7, 74, "Input"],
Cell[47084, 1432, 542, 8, 75, "Message"],
Cell[47629, 1442, 333, 5, 18, "Message"],
Cell[47965, 1449, 543, 15, 41, "Output"]
}, Open  ]],
Cell[48523, 1467, 312, 8, 41, "Text"],
Cell[48838, 1477, 1663, 40, 146, "Input"],
Cell[50504, 1519, 494, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[51023, 1539, 492, 10, 74, "Input"],
Cell[51518, 1551, 758, 20, 57, "Output"]
}, Open  ]],
Cell[52291, 1574, 47, 0, 24, "Text"],

Cell[CellGroupData[{
Cell[52363, 1578, 179, 4, 58, "Input"],
Cell[52545, 1584, 24623, 860, 184, 9996, 674, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[77241, 2452, 52, 1, 67, "Section",
  CellTags->"s:3"],
Cell[77296, 2455, 390, 10, 56, "Text"],
Cell[77689, 2467, 894, 26, 156, "DefinitionBox3Col"],
Cell[78586, 2495, 98, 4, 16, "Caption"],
Cell[78687, 2501, 1063, 31, 198, "DefinitionBox3Col"],
Cell[79753, 2534, 95, 4, 16, "Caption"]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

