(* :Author: Chris Williamson *)

BeginPackage["WebServices`"];

WebServices`Information`$Version = "Web Services Version 1.0.2 (June 2005)";

WebServices`Information`$VersionNumber = 1.0;

WebServices`Information`$ReleaseNumber = 2;

(* Functions *)

InstallService::usage=
  "InstallService[wsdlURL] installs the web service operations specified
  in the WSDL URL into a default context defined by the service name and
  port name of the service operation.
  InstallService[wsdlURL, myContext] installs the web service operations
  specified in the WSDL URL into a context specified by the second parameter.";

InstallServiceOperation::usage=
  "InstallServiceOperation[symbolName, endPoint, argumentDefs, headerDefs]
  creates a function for the web service operation defined by the end point,
  argument definitions, and header definitions.";

UninstallService::usage=
  "UninstallService[context] uninstalls all web service operations in the
  context provided.";

UninstallServiceOperation::usage=
  "UninstallServiceOperation[symbol] uninstalls the web service operation defined
  as the symbol.  The OperationPalette is cleared and the symbol is removed.";
    
OperationPalette::usage=
  "OperationPalette[symbol] displays a palette that can be used to understand a
  web service operation.  The palette displays documentation, the function signature,
  type information, and buttons which allow the user to paste useful templates.";

ToServiceRequest::usage=
  "ToServiceRequest[parameters, headerParameters] builds a request message using
  input provided in the parameters.  ToServiceRequest[symbol, parameters___] builds
  the request message for a web service operation using input provided in the parameters
  and information linked with the symbol.";

InvokeServiceOperation::usage=
  "InvokeServiceOperation[endPointURL, requestMessage]  invokes a web service operation
  using the request message.  The message is sent to the end point specified in the
  first argument.  The response message is returned.
  InvokeServiceOperation[symbol, parameters___] invokes a web service operation using
  input provided in the parameters and information linked with the symbol.  The
  response message is returned.
  InvokeServiceOperation[symbol, requestMessage] invokes a web service operation using
  the request message and information linked with the symbol.  The response message is
  returned.";

FromServiceResponse::usage=
  "FromServiceResponse[response] converts a response message into a useful Mathematica
  expression.  For SOAP messages, the SOAP envelope and SOAP body tags are stripped
  from the results.  The values are then converted into Rule syntax.";
  
ServicesImpl::usage="ServicesImpl[name] represents a web services implemenation.";

(* Global Options *)

$ServicesImpl::usage =  
  "$ServiceImpl specifies the default implementation that will be used to invoke Web Services.
  This should be set to the ServicesImpl object that corresponds with the Web Services
  implementation."

$InstalledServices::usage=
  "$InstalledServices is a list of the web service operations installed."

$PrintServiceRequest::usage=
  "$PrintServiceRequest uses the Mathematica Print function to print the message sent to a
  web service.  The default is False."

$PrintServiceResponse::usage=
  "$PrintServiceResponse uses the Mathematica Print function to print the message received
  from a web service before it is deserialized into a Rule syntax expression.  The default
  is False."
  
$PrintWSDLDebug::usage=
  "$PrintWSDLDebug specifies whether WSDL debugging information will be printed when installing
  a web service.  The default is False.";

$PrintShortErrorMessages::usage=
  "$PrintShortErrorMessages specifies whether error messages will be shortened for the user
  to avoid long intimidating error messages.  The default is True."  

Begin["`Client`Private`"]

$ServicesImpl = ServicesImpl["Java"];

$InstalledServices = {};

$PrintServiceRequest = False;

$PrintServiceResponse = False;
  
$PrintShortErrorMessages = True;

$PrintWSDLDebug = False;
    
Options[ToServiceRequest] =
  {   
    (* SOAP Options *)
    "OperationName" -> Automatic,
    "OperationStyle" -> Automatic,
    "SOAPActionURI" -> Automatic,
    "TransportStyleURI" -> Automatic,
    "EncodingStyle" -> Automatic,
    "EncodingStyleURI" -> Automatic,
    "EncodingStyle" -> Automatic,
    "Username" -> Automatic,
    "Password" -> Automatic,
    "ExpressionWrapperFunction" -> Identity,
    "DeencodeParameters" -> False
  }

Options[FromServiceResponse] = 
  {
    (* General Options *)
    "LongForm" -> False,
    
    (* SOAP Options *)
    "ReturnType" -> Automatic
  }

Options[InvokeServiceOperation] = 
  Join[
    Options[ToServiceRequest], 
    Options[FromServiceResponse], 
    {
      "Timeout"->Automatic
    }
  ]

Options[InstallServiceOperation] = 
  Join[
    {
      "AllowShortContext" -> True
    },
    Options[InvokeServiceOperation]
  ]
  
Options[InstallService] = 
  {
    "LongForm" -> False,
    "AllowShortContext" -> True
  }
      
InstallService[wsdlURL_String, options___?OptionQ] := 
  InstallService[$ServicesImpl, wsdlURL, options]

InstallService[wsdlURL_String, context_String, options___?OptionQ] := 
  InstallService[$ServicesImpl, wsdlURL, context, options]

InstallServiceOperation[exprName_Symbol, 
                        endPoint_String,
                        args_List,
                        header_List,
                        options___?OptionQ] :=
  InstallServiceOperation[$ServicesImpl, exprName, endPoint, args, header, options]
  
UninstallServiceOperation[operation_Symbol] := UninstallServiceOperation[$ServicesImpl, operation]

UninstallService[context_String] := UninstallService[$ServicesImpl, context]

OperationPalette[operation_Symbol] := OperationPalette[$ServicesImpl, operation]

ToServiceRequest[parameters_List, 
                 headerParameters_List, 
                 options___?OptionQ] :=
  ToServiceRequest[$ServicesImpl, parameters, headerParameters, options];

ToServiceRequest[x___] := (
  $Failed
);

InvokeServiceOperation[endPoint_String, 
                       message:XMLObject["Document"][___], 
                       options___?OptionQ] :=
  InvokeServiceOperation[$ServicesImpl, endPoint, message, options]


FromServiceResponse[response:XMLObject["Document"][{___}, _XMLElement, ___],
                        options___?OptionQ] :=
  FromServiceResponse[$ServicesImpl, response, options];  

FromServiceResponse[x___] := (
  Message[InvokeServiceOperation::"rspnsdoc", x];
  $Failed
);

(* Set package directory used to find implementation files *)
$webservicesPackageDirectory = DirectoryName[System`Private`FindFile[$Input]];

End[];

Begin["`Package`"];

$PrintPerformanceNumbers = False;

SetAttributes[ canonicalOptions, {Listable}];
canonicalOptions[name_Symbol -> val_] := SymbolName[name] -> val;
canonicalOptions[expr___] := expr;

(* Set package directory used to find implementation files *)
$webservicesPackageDirectory = DirectoryName[System`Private`FindFile[$Input]];

End[]; (* Database`Package` *)

(* Make the Package` symbols visible to all implementation files as they are read in. *)
AppendTo[$ContextPath, "WebServices`Package`"]

(* Read in Security files *)
If[WebServices`Package`$webServicesSecurityInitialized =!= True, 
  Get[ToFileName[{$webservicesPackageDirectory, "Kernel"}, "Security.m"]]
];

(* Read in the Implementation Files  *)
(
  Get[ToFileName[#, "SchemaUtilities.m"]];
  Get[ToFileName[#, "Utilities.m"]];
  Get[ToFileName[#, "OperationPalette.m"]];
  Get[ToFileName[#, "WSDL.m"]];
  Get[ToFileName[#, "Normalize.m"]];
  Get[ToFileName[#, "Deserialize.m"]];
  Get[ToFileName[#, "Serialize.m"]];
  Get[ToFileName[#, "JavaImpl.m"]];
)& @ ToFileName[$webservicesPackageDirectory, "Kernel"]

EndPackage[];