(* :Author: Chris Williamson *)

Begin["`Package`"];

Deserialize::usage = 
  "Deserialize is a utility function used to deserialize a web service response into
  a format desired by the user.";

End[];
  
Begin["`Deserialize`Private`"]

InvokeServiceOperation::"desrlz"= "Could not deserialize: `1`"
InvokeServiceOperation::"href"= "Href not found."
InvokeServiceOperation::"insecure" = "This expression is insecure: `1`"

base64 = Null;

base64Decode[str_String] := (
  If[base64 === Null, base64 = JavaNew["org.apache.axis.encoding.Base64"]];
  JavaObjectToExpression[JavaNew["java.lang.String", base64@decode[str], 0]]
);

(* Deserialize an Expression *)

Deserialize[
  ServicesImpl["Java"],
  m:XMLElement[
    name:(_String | {_String, _String}), 
    attrs:{___}, 
    {value_XMLElement}], 
  {"http://www.wolfram.com/XML/", "Expression"}, 
  _List,
  options___?OptionQ] :=
  Module[{prefix, val, longForm, returnName, wrapperFunction, 
          opts = canonicalOptions[Flatten[{options}]]},
    (* Get options *)
    {longForm, returnName, wrapperFunction} = 
      {"LongForm", "ReturnName", "ExpressionWrapperFunction"} /. 
        opts  /. Options[InvokeServiceOperation];

    (* determine whether to return the name and/or type with the value *)
    If[longForm === True && returnName =!= False, 
      prefix = {name, attrs}, 
      If[returnName =!= False, 
        If[ListQ[name], prefix = Last[name], prefix = name], 
        prefix = Null
      ];
    ];

    val = value /. 
      XMLElement[n:(_String | {_String, _String}), _List, val_List] -> XMLElement[n, {}, val];
            
    val = XMLElement[{"http://www.wolfram.com/XML/", "Expression"}, {}, {
      XMLElement[{"http://www.wolfram.com/XML/", "Function"}, {}, 
        {XMLElement[{"http://www.wolfram.com/XML/", "Symbol"}, {}, {"HoldComplete"}], val}]}];
        
    val = XML`NotebookML`SymbolicExpressionMLToExpression[val];
    If[InsecureExprQ[val], 
      Message[InvokeServiceOperation::"insecure", val];
      val = $Failed, 
      val = Function[x, wrapperFunction @@ x , {HoldAllComplete}] [ val];
    ];

    (* Return value *)
    If[MatchQ[prefix, Null], 
      val,
      prefix->val
    ]
  ];

Deserialize[
  ServicesImpl["Java"],
  m:XMLElement[
    name:(_String | {_String, _String}), 
    attrs:{___}, 
    {value_}], 
  {"http://www.w3.org/1998/Math/MathML", ("math" | "math.type")}, 
  _List,
  options___?OptionQ] :=
  Module[{prefix, val, longForm, returnName = False, 
          opts = canonicalOptions[Flatten[{options}]]},
    (* Get options *)
    {longForm, returnName} = {"LongForm", "ReturnName"} /. 
        opts  /. Options[InvokeServiceOperation];

    (* determine whether to return the name and/or type with the value *)
    If[longForm === True && returnName =!= False, 
      prefix = {name, attrs}, 
      If[returnName =!= False, 
        If[ListQ[name], prefix = Last[name], prefix = name], 
        prefix = Null
      ];
    ];

    val = value /. 
      XMLElement[n:(_String | {_String, _String}), _List, val_List] -> XMLElement[n, {}, val];
    
    val = XML`ToCompactXML[XMLElement[{"http://www.w3.org/1998/Math/MathML", "math"}, {}, {val}]];
    
    (* Return value *)
    If[MatchQ[prefix, Null], 
      val,
      prefix->val
    ]
  ];

(* Deserialize the base types *)
Deserialize[
  ServicesImpl["Java"],
  XMLElement[
    name:(_String | {_String, _String}), 
    attrs:{___}, 
    {value_}], 
  {_?BaseTypeNamespaceQ, type_?BaseTypeQ}, 
  _List,
  options___?OptionQ] :=
  Module[{prefix, val, longForm, returnName, 
          opts = canonicalOptions[Flatten[{options}]]}, 
  
    (* Get options *)
    {longForm, returnName} = 
      {"LongForm", "ReturnName"} /. 
        opts  /. Options[InvokeServiceOperation];

    (* determine whether to return the name and/or type with the value *)
    If[longForm === True && returnName =!= False, 
      prefix = {name, attrs}, 
      If[returnName =!= False, 
        If[ListQ[name], prefix = Last[name], prefix = name], 
        prefix = Null
      ];
    ];
        
    (* Do the conversion *)
    Which[ 
      StringMatchQ[type, "boolean"],  
        If[StringMatchQ[value, "True", IgnoreCase -> True], val = True, val = False], 
      StringMatchQ[type, "double"] || 
        StringMatchQ[type, "float"] || 
        StringMatchQ[type, "int"] || 
        StringMatchQ[type, "integer"] || 
        StringMatchQ[type, "long"] || 
        StringMatchQ[type, "short"] || 
        StringMatchQ[type, "decimal"] || 
        StringMatchQ[type, "nonPositiveInteger"] || 
        StringMatchQ[type, "negativeInteger"] || 
        StringMatchQ[type, "nonNegativeInteger"] || 
        StringMatchQ[type, "unsignedLong"] || 
        StringMatchQ[type, "positiveInteger"] || 
        StringMatchQ[type, "unsignedInt"] || 
        StringMatchQ[type, "unsignedShort"] || 
        StringMatchQ[type, "unsignedByte"], 
        val = ToExpression[value, InputForm, HoldComplete];
        If[NumberQ[val[[1]]], 
          val = ReleaseHold[val], 
          Message[InvokeServiceOperation::"insecure", val];
          val = $Failed
        ],
      StringMatchQ[type, "byte"] && DigitQ[value],
        val = ToExpression[value, InputForm, HoldComplete];  
        If[NumberQ[val[[1]]], 
          val = ReleaseHold[val];
          If[val < 0, val = val + 255],
          Message[Deserialize::blah, value];
          val = $Failed
        ],
      StringMatchQ[type, "base64Binary"], val = base64Decode[value],
      StringMatchQ[type, "base64"], val = base64Decode[value],
      True, val = value
    ];

    (* Return value *)
    If[MatchQ[prefix, Null], 
      val,
      prefix->val
    ]
  ]

(* Deserialize Array XMLElement *)
Deserialize[
  ServicesImpl["Java"],
  XMLElement[
    name:(_String | {_String, _String}), 
    attrs:{___, {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->arrayType_, ___}, 
    children:{___}],
  (type:{_String, _String} | type:{}), 
  namespaces_List,
  options___?OptionQ] :=
  Module[{longForm, returnName, nspaces, prefix,
          namespace = "", t = Null, schema, value, n, 
          typeSchema = Null, schemaNamespaces = {}, typeNS = Null,
          opts = canonicalOptions[Flatten[{options}]]},
    
    (* Get options and add namespaces *)
    {longForm, returnName} = 
      {"LongForm", "ReturnName"} /. 
        opts  /. Options[InvokeServiceOperation]; 
    
    nspaces = MapNamespaces[m, namespaces];

    (* determine whether to return the name and/or type with the value *)
    If[longForm === True && returnName =!= False, 
      prefix = {name, attrs}, 
      If[returnName =!= False, 
        If[ListQ[name], prefix = Last[name], prefix = name], 
        prefix = Null
      ];
    ];
    
    opts = DeleteCases[opts, "ReturnName"->_];
    
    If[MatchQ[type, {_String, _String}], namespace  = First[type]];
    
    If[StringQ[arrayType] && StringMatchQ[arrayType, "*[*"], 
      t = StringTake[arrayType, {1, First[First[StringPosition[arrayType, "["]]]-1}];
    ];
        
    If[MatchQ[arrayType, {_String, _String}],     
      If[StringMatchQ[Last[arrayType], "*[*"], 
        t = {First[arrayType], StringTake[Last[arrayType], {1, First[First[StringPosition[Last[arrayType], "["]]]-1}]};
      ];
    ];
    
    t = GetType[t, nspaces];   
        
    (* If an array *)
    AppendTo[opts, "ReturnName"->False];
    value = (Deserialize[ServicesImpl["Java"], #, t, nspaces, opts] & /@ children);

    If[children === {} && type =!= {"http://schemas.xmlsoap.org/soap/encoding/","array"}, 
      value = Null
    ];
                   
    (* Return value *)
    If[MatchQ[prefix, Null], 
      value,
      prefix->value
    ]
  ]

resolveReference[
  m:XMLElement[
    name:(_String | {_String, _String}), 
    {___, {"", "href"}->id_String,___}, 
    {___}],
  namespaces_List,
  options___?OptionQ] :=
  Module[{elmnt, t = Null, nspaces, d, longForm, returnName,
          opts = canonicalOptions[Flatten[options]], prefix},
    elmnt = GetHref[StringDrop[id, 1]];
    If[elmnt === Null, 
      Message[InvokeServiceOperation::"href"];
      Null,
      nspaces = MapNamespaces[elmnt, namespaces];
      t = GetType[elmnt, nspaces];
      d = Deserialize[ServicesImpl["Java"], elmnt, t, nspaces, options];

      (* Get options and add namespaces *)
      {longForm, returnName} = 
        {"LongForm", "ReturnName"} /. 
           opts  /. Options[InvokeServiceOperation]; 
    
      (* determine whether to return the name and/or type with the value *)
      If[longForm === True && returnName =!= False, 
        prefix = {name, GetAttributes[d]}, 
        If[returnName =!= False, 
          If[ListQ[name], prefix = Last[name], prefix = name], 
          prefix = Null
        ];
      ];
        
      (* Return value *)
      If[MatchQ[prefix, Null], 
        d,
        prefix->Last[d]
      ]
    ]
  ];

resolveReference[x___] := Null;

deserializeChild[child_, nspaces_List, typeSchema_, typeNS_, typeNspaces_, df_, opts_] := 
  Module[{name, childNS = typeNS, childSchema, childNspaces = typeNspaces, d, defaultForm, type = {}, typeName},
    name = GetName[child];
    Switch[name,
      _String | {"", _String},
        If[MatchQ[name, {"", _String}], name = Last[name]];
        childSchema = GetElementDef[name, typeSchema];
        childNspaces = MapNamespaces[childSchema, typeNspaces];
        childNS = typeNS,
      {_String, _String},
        If[typeNS =!= Null && StringMatchQ[typeNS, First[name]],
          childSchema = GetElementDef[Last[name], typeSchema];
          childNspaces = MapNamespaces[childSchema, typeNspaces];
          childNS = typeNS;
          If[childSchema == Null,
            {name, childSchema, childNspaces, defaultForm} = GetElementContent[name]
          ],
          {name, childSchema, childNspaces, defaultForm} = GetElementContent[name];
          childNS = First[name];
        ],
      _, 
        If[IsArray[typeSchema],
          childSchema = GetElementDef[Null, typeSchema];
          childNspaces = MapNamespaces[childSchema, typeNspaces];
          childNS = typeNS, 
          childSchema = Null;
          childNspaces = Null;
          childNS = {};
        ] 
    ];
    eType = GetElementType[childSchema];
    If[StringQ[eType],
      type = GetType[eType, childNspaces, childNS],
      If[eType =!= Null, 
        typeName = GetTypeName[eType];
        If[typeName == Null, 
          type = {}, 
          type = {childNS, typeName} 
        ];        
      ];
    ];
    d = Deserialize[ServicesImpl["Java"], child, type, nspaces, opts];
    d
  ]

(* Default Deserialize XMLElement *)
Deserialize[
  ServicesImpl["Java"],
  m:XMLElement[
    name:(_String | {_String, _String}), 
    attrs:{___}, 
    children:{___}],
  (type:{_String, _String} | type:{}), 
  namespaces_List,
  options___?OptionQ] :=
  Module[{longForm, nspaces, prefix, typeNS = Null, t, returnName,
          schema = Null, typeSchema = Null, value, n, defaultForm,
          schemaNamespaces = {}, opts = canonicalOptions[Flatten[{options}]], 
          ref},
    
    ref = resolveReference[m, namespaces, opts];
    If[ref =!= Null,
      Return[ref];
    ];
   
    (* Get options and add namespaces *)
    {longForm, returnName} = 
      {"LongForm", "ReturnName"} /. 
        opts  /. Options[InvokeServiceOperation]; 
    
    nspaces = MapNamespaces[m, namespaces];

    (* determine whether to return the name and/or type with the value *)
    If[longForm === True && returnName =!= False, 
      prefix = {name, attrs}, 
      If[returnName =!= False, 
        If[ListQ[name], prefix = Last[name], prefix = name], 
        prefix = Null
      ];
    ];
    
    opts = DeleteCases[opts, "ReturnName"->_];
    
    {t, typeSchema, schemaNamespaces, defaultForm} = GetTypeContent[type];
    If[MatchQ[type, {_String, _String}], typeNS = First[type]];
    
    If[IsArray[typeSchema],
      AppendTo[opts, "ReturnName"->False];
    ];
    
    (* Do the conversion *)
    If[MatchQ[children, {_String}],
      value = First[children],
      (* If a compound object *)
      value = (deserializeChild[#, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, opts] & /@ children);
    ];

    If[children === {} && 
       type =!= {"http://schemas.xmlsoap.org/soap/encoding/","array"} && 
       !IsArray[typeSchema], 
      value = Null
    ];
                   
    (* Return value *)
    If[MatchQ[prefix, Null], 
      value,
      prefix->value
    ]
  ]

Deserialize[_ServicesImpl, x___] := (
  Message[InvokeServiceOperation::"desrlz", {x}];
  $Failed 
);

  
End[]
