(* :Author: Chris Williamson *)

Begin["`Package`"];

MapReferences::usage =
  "MapReferences is a utility function that is used to map references in a SOAP envelope to
  Mathematica symbol.";

GetHref::usage = 
  "GetHref is a utility function that is used to retrieve references from a SOAP envelope.";

End[];

Begin["`JavaImpl`Private`"]

InvokeServiceOperation::"native" = 
  "An error occurred: `1`"

InvokeServiceOperation::"env" = 
  "Invalid SOAP envelope: `1`"

InvokeServiceOperation::"multrtn" = 
  "Multiple return values are not supported"

InvokeServiceOperation::"prtcl" = 
  "Incorrect protocol used in end point URL: `1`.  Should be: `2`"

InvokeServiceOperation::"opttype" = 
  "`1` is not the correct type for Option `2` "

InvokeServiceOperation::"opstyle" = 
  "OperationStyle must be set to 'rpc' or 'document'."

InvokeServiceOperation::"rspns" = 
  "Invoke Failed.  Response is $Failed or Null."

InvokeServiceOperation::"rspnsdoc" = 
  "Response cannot be parsed:\n `1`"

InvokeServiceOperation::"rspnsenv" = 
  "SOAP Envelope cannot be parsed from response:\n `1`"

InvokeServiceOperation::"rspnsbdy" = 
  "SOAP Body cannot be parsed from response:\n `1`"

InvokeServiceOperation::"rspnsrtn" = 
  "SOAP Return Type cannot be parsed from response:\n `1`"

InvokeServiceOperation::"rspnsinv" = 
  "SOAP Return element invalid:\n `1`"

InvokeServiceOperation::"rspnsflt" = "SOAPFault occurred: `1`"

InvokeServiceOperation::"mapparam" = 
  "Could not map parameters: \n`1`,\n`2`"
  
Needs[ "JLink`" ];

(* InvokeServiceOperation *)

call = Null;

$webServicesDirectory = DirectoryName[ System`Private`FindFile[ $Input]];

AppendTo[$Path, $webServicesDirectory <> "/conf"];

$securityConfiguration = System`Private`FindFile[ "SecurityConfiguration.m"];

If[$securityConfiguration === $Failed,
  SetSecurity[],
  SetSecurity[DirectoryName[$securityConfiguration], "SecurityConfiguration.m"]
];

invoke[endPoint_String, env_, options___?OptionQ] :=
  Module[{service, url, response, transportURI, operationStyle, 
          soapActionURI, username, password, timeout,
          opts = canonicalOptions[Flatten[{options}]]},
  
    If[call === Null || !JavaObjectQ[call], 
      service = JavaNew["org.apache.axis.client.Service"];
      call = service@createCall[];
    ];
        
    (* Process options *)
    {transportURI, operationStyle, soapActionURI, username, password, timeout} = 
      {"TransportStyleURI", "OperationStyle", "SOAPActionURI", 
       "Username", "Password", "Timeout"} /. 
         opts  /. Options[InvokeServiceOperation];

    (* Add Transport URI *)

    (* Check to make sure the url contains the correct protocol *)
    If[!StringMatchQ[endPoint, "http://*"], 
      Message[InvokeServiceOperation::"prtcl", endPoint, "http"];
      Return[$Failed],
      (* Catch Exception *) 
      url = JavaNew["java.net.URL", endPoint];
      If[url === $Failed, Return[$Failed]];
      call@setTargetEndpointAddress[ url];
    ];
    
    If[soapActionURI =!= Automatic,  
      If[StringQ[soapActionURI], 
        call@setSOAPActionURI[soapActionURI],
        Message[InvokeServiceOperation::"opttype", 
                soapActionURI,
                "SOAPActionURI"];       
      ];
    ];
                    
    If[operationStyle =!= Automatic, 
      If[StringQ[operationStyle],
        call@setOperationStyle[operationStyle],
        Message[InvokeServiceOperation::"opttype", 
                operationStyle, 
                "OperationStyle"];
        Return[$Failed];
      ],
      operationStyle = "rpc"
    ];

    If[username =!= Automatic, 
      If[StringQ[username],
        call@setUsername[username],
        Message[InvokeServiceOperation::"opttype", 
                username, 
                "Username"];
        Return[$Failed];
      ];
    ];

    If[password =!= Automatic, 
      If[StringQ[password],
        call@setPassword[password],
        Message[InvokeServiceOperation::"opttype", 
                password, 
                "Password"];
        Return[$Failed];
      ];
    ];
    
    If[timeout =!= Automatic, 
      If[IntegerQ[timeout],
        call@setTimeout[JavaNew["java.lang.Integer", timeout]],
        Message[InvokeServiceOperation::"opttype", 
                timeout, 
                "Timeout"];
        Return[$Failed];
      ];
    ];
            
    If[TrueQ[$PrintServiceRequest], Print[env@toString[]]];
    
    proxyHost = System`getProperties[]@getProperty["proxyHost"];
    proxyPort = System`getProperties[]@getProperty["proxyPort"];
    If[proxyHost =!= Null, 
      AxisProperties`setProperty["http.proxyHost", proxyHost];
    ];
    If[proxyPort =!= Null, 
      AxisProperties`setProperty["http.proxyPort", proxyPort];
    ];

    (* invoke call *)
    If[TrueQ[$PrintPerformanceNumbers], Print["Invoking: ",Date[]]];

    response = 
      If[operationStyle === Automatic || 
         StringMatchQ[operationStyle, "rpc"] ||
         StringMatchQ[operationStyle, "document"],
        call@invoke[ env],
        Message[InvokeServiceOperation::"opstyle"];
        Return[$Failed];
      ];

    If[TrueQ[$PrintPerformanceNumbers], Print["Finished Invoking: ",Date[]]];

    If[response =!= $Failed && TrueQ[$PrintServiceResponse], Print[response@toString[]]];
        
    (* process response *)
    If[response =!= $Failed && response =!= Null, 
      response = XML`ToVerboseXML[XML`Parser`XMLGetString[response@toString[]]],
      Message[InvokeServiceOperation::"rspns"];
      Return[$Failed];
    ];
    
    response
  ]

MapReferences[ServicesImpl["Java"], m_XMLElement] := 
  Module[{cases},
    cases = 
      Cases[m, 
        e:XMLElement[
          (_String | {_String, _String}), 
          {___, {"", "id"}->id_String, ___}, 
          {___}] :> {id, e}, Infinity];
    (GetHref[First[#]] = Last[#]) & /@ cases;
  ]

FromServiceResponse[
  ServicesImpl["Java"],
  response:XMLObject["Document"][{___}, _XMLElement, ___],
  options___?OptionQ] :=
  Module[{envelope, namespaces, body, return, root, type, returnType,
          resp, opts = canonicalOptions[Flatten[{options}]]},
    
    {returnType} = 
      {"ReturnType"} /. 
         opts  /. Options[FromServiceResponse]; 
               
     If[returnType =!= Automatic, 
      If[!MatchQ[returnType, {_String, _String}], 
        Message[InvokeServiceOperation::"opttype", 
                returnType, 
                "ReturnType"];
        returnType = Automatic;
      ];
    ];
    
    (* Process SOAP Envelope *)
    If[MatchQ[
         response[[2]], 
         XMLElement[
         ("Envelope" | {"http://schemas.xmlsoap.org/soap/envelope/", "Envelope"}), 
         {___}, 
         {__XMLElement}]],
      envelope = response[[2]], 
      Message[InvokeServiceOperation::"rspnsenv", 
              response[[2]]];
      Return[$Failed]
    ];
    namespaces = MapNamespaces[envelope, {}];
    
    (* Process SOAP Body *)
    body = 
      Cases[
        envelope[[3]], 
        XMLElement[
          ("Body" | {"http://schemas.xmlsoap.org/soap/envelope/", "Body"}), 
          {___}, 
          {__XMLElement}], 
        1];
    If[Length[body] > 0, 
      body = First[body], 
      Message[InvokeServiceOperation::"rspnsbdy", 
              ExportString[envelope[[3]], "XML"]];
      Return[$Failed]
    ];
    namespaces = MapNamespaces[body, namespaces];

    (* Process return *)
    return = 
      Cases[
        body[[3]], 
        XMLElement[__, {___}, {___XMLElement}, ___],
        1];
    If[Length[return] > 0, 
      return = First[return], 
      Message[InvokeServiceOperation::"rspnsrtn", 
              ExportString[body[[3]], "XML"]];
      Return[$Failed]
    ];
    namespaces = MapNamespaces[return, namespaces];
    
    (* Process SOAPFault *)
    (*
    fault = 
      Cases[
        body[[3]], 
        XMLElement[
          ("Fault" | { "http://schemas.xmlsoap.org/soap/envelope/", "Fault"}), 
          {___}, 
          {__XMLElement}],
        1];
    If[Length[fault] > 0, 
      fault = First[fault];
      Message[InvokeServiceOperation::"rspnsflt"];
      Return[$Failed]
    ];
    *)
    
    If[MatchQ[return[[3]], {___XMLElement}],
      Switch[Length[return[[3]]], 
        0, Return[], 
        1, resp = return[[3,1]],
        _, Message[InvokeServiceOperation::"multrtn"];Return[$Failed]
      ], 
      Message[InvokeServiceOperation::"rspnsinv", return[[3]]]; 
      Return[$Failed]
    ];
    
    root = body;
    
    (* Deserialize *)
    If[TrueQ[$PrintPerformanceNumbers], Print["Deserializing: ", Date[]]];

    GetHref[x_] := Null;
    
    MapReferences[ServicesImpl["Java"], root];

    namespaces = MapNamespaces[resp, namespaces];
    type = GetType[resp, namespaces, returnType];
    
    AppendTo[opts, "ReturnName"->False];

    resp = Deserialize[ServicesImpl["Java"], resp, type, namespaces, opts];
 
    Clear[GetHref];

    If[TrueQ[$PrintPerformanceNumbers], Print["Finished Deserializing: ", Date[]]];

    resp
  ];

removeEncoding[p:({(_String | {_String, _String}), {___}}->value_)] :=
  Module[{x},
    x = removeType[p];
    removeArrayType[x]
  ];

removeEncoding[p___] := p;

removeType[({name:(_String | {_String, _String}), 
               params:{___,{_String?SchemaInstanceNamespaceQ, "type"}->_,___}}->value_)] :=
  {name, DeleteCases[params, {_String?SchemaInstanceNamespaceQ, "type"} -> _ ]}->value;

removeType[p___] := p;

removeArrayType[({name:(_String | {_String, _String}), 
                 params:{___,{"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->_,___}}->value_)] :=
  {name, DeleteCases[params, {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->_ ]}->value;

removeArrayType[p___] := p;
  
ToServiceRequest[ServicesImpl["Java"],
                  parameters_List, 
                  headerParameters_List, 
                  options___?OptionQ] :=
  Module[{encodingStyleURI, encodingStyle, namespaces, 
          params, failed, operationAttrs = {}, es = "", operation, type,
          headerAttrs = {}, headerParams, children, envelope, deencodeParameters,
          ns = Null, name = Null, n, a},
          
    NamespaceId = 1;
    
    (* Process options *)
    {encodingStyleURI, encodingStyle, operationName, deencodeParameters} = 
      {"EncodingStyleURI", "EncodingStyle", "OperationName", "DeencodeParameters"} /. 
         canonicalOptions[Flatten[{options}]]  /. Options[ToServiceRequest];

    If[TrueQ[$PrintPerformanceNumbers], Print["Building SOAPEnvelope: ", Date[]]];

    If[MatchQ[operationName, {_String, _String}],
      ns = First[operationName];
      name = Last[operationName],
      If[operationName =!= Automatic, 
        Message[InvokeServiceOperation::"opttype", 
                soapActionURI,
                "SOAPActionURI"];
      ];
    ];
  
    namespaces = {{"http://www.w3.org/2001/xmlns/", "soapenv"}->"http://schemas.xmlsoap.org/soap/envelope/",
                  {"http://www.w3.org/2001/xmlns/", "soapenc"}->"http://schemas.xmlsoap.org/soap/encoding/",
                  {"http://www.w3.org/2001/xmlns/", "xsd"}->"http://www.w3.org/2001/XMLSchema",
                  {"http://www.w3.org/2001/xmlns/", "xsi"}->"http://www.w3.org/2001/XMLSchema-instance",
                  {"http://www.w3.org/1999/xmlns/", "xmlns"}->"http://www.w3.org/1999/xmlns/",
                  {"http://www.w3.org/2000/xmlns/", "xmlns"}->"http://www.w3.org/2000/xmlns/",
                  {"http://www.w3.org/2001/xmlns/", "xmlns"}->"http://www.w3.org/2001/xmlns/"};

    If[encodingStyleURI =!= Automatic && encodingStyleURI =!= {}, 
      If[MatchQ[encodingStyleURI, {___String}], 
        es = StringReplace[ExportString[encodingStyleURI, "Words"], {" "->"", "\n"->""}];
        AppendTo[headerAttrs, {"soapenv", "encodingStyle"}->es],
        Message[InvokeServiceOperation::"opttype", 
                encodingStyleURI, 
                "EncodingStyleURI"];
      ];
    ];
    headerParams = 
      ((type = GetType[#, namespaces];
        If[TrueQ[deencodeParameters],n=removeEncoding[#],n=#];
        n = 
          Normalize[
            ServicesImpl["Java"], 
            n, 
            type, 
            Null, 
            namespaces,
            False,
            encodingStyle,
            encodingStyleURI];
        Serialize[ServicesImpl["Java"], n, type, namespaces, options]) & /@ headerParameters);
    failed = Cases[headerParams, $Failed, Infinity];
    If[Length[failed] > 0, Return[$Failed]];
    
    If[ns =!= Null, AppendTo[namespaces, {"http://www.w3.org/2001/xmlns/", "ns0"}->ns]];
    (* serialize parameters *)    
    params = 
      ((type = GetType[#, namespaces];
        If[TrueQ[deencodeParameters],n=removeEncoding[#],n=#];
        n = 
          Normalize[
            ServicesImpl["Java"], 
            n, 
            type, 
            Null, 
            namespaces,
            False, 
            encodingStyle,
            encodingStyleURI];
        Serialize[ServicesImpl["Java"], n, type, namespaces, options]) & /@ parameters);
    failed = Cases[params, $Failed, Infinity];
    If[Length[failed] > 0, Return[$Failed]];

    (* Add operation *)
    If[encodingStyleURI =!= Automatic && encodingStyleURI =!= {}, 
      If[MatchQ[encodingStyleURI, {___String}], 
        es = StringReplace[ExportString[encodingStyleURI, "Words"], {" "->"", "\n"->""}];
        AppendTo[operationAttrs, {"soapenv", "encodingStyle"}->es],
        Message[InvokeServiceOperation::"opttype", 
                encodingStyleURI, 
                "EncodingStyleURI"];
      ];
    ];
    
    If[ns =!= Null && name =!= Null, 
      AppendTo[operationAttrs, {"xmlns", "ns0"}->ns];
      children = {XMLElement[{"ns0", name}, operationAttrs, params]},
      children = params;
    ];    
    
    children = {XMLElement[{"soapenv", "Body"}, {}, children]};
    If[headerParams =!= {}, 
      PrependTo[children, XMLElement[{"soapenv", "Header"}, headerAttrs, headerParams]]
    ];
    
    (* Create envelope *)
    envelope = 
      XMLObject["Document"][
        {}, 
        XMLElement[
          {"soapenv", "Envelope"}, 
          {{"xmlns", "soapenv"}->"http://schemas.xmlsoap.org/soap/envelope/",
           {"xmlns", "xsd"}->"http://www.w3.org/2001/XMLSchema",
           {"xmlns", "xsi"}->"http://www.w3.org/2001/XMLSchema-instance", 
           {"xmlns", "soapenc"}->"http://schemas.xmlsoap.org/soap/encoding/"},
          children],
        {}];
    
    If[TrueQ[$PrintPerformanceNumbers], Print["Finished Building SOAPEnvelope: ", Date[]]];
    
    envelope
  ]

convertMessageToJavaObject[message:XMLObject["Document"][___]] :=
  Module[{xml, is, envelope},
    (* convert to a string *)
    xml = ExportString[message, "XML"];
    If[xml === $Failed, 
      Message[InvokeServiceOperation::"env", message];
      Return[$Failed]
    ];
      
    (* convert to an InputStream *)
    is = JavaNew["java.io.ByteArrayInputStream", 
                 MakeJavaObject[xml]@getBytes[]];
    If[is === $Failed, 
      Message[InvokeServiceOperation::"env", xml];
      Return[$Failed]
    ];
  
    (* convert to a SOAPEnvelope object *)
    envelope = JavaNew["org.apache.axis.message.SOAPEnvelope", is];
    If[envelope === $Failed, 
      Message[InvokeServiceOperation::"env", xml];
      Return[$Failed]
    ];
    envelope
  ]
  
InvokeServiceOperation[ServicesImpl["Java"],
                       endPoint_String, 
                       message:XMLObject["Document"][___], 
                       options___?OptionQ] :=
  JavaBlock[
    Module[{envelope, messages, result},

      InstallJava[];
      
      Block[{JLink`Exceptions`Private`reportException},
      
        If[TrueQ[$PrintShortErrorMessages], 
          JLink`Exceptions`Private`reportException = reportShortError,
          JLink`Exceptions`Private`reportException = reportError
        ];
        
        envelope = convertMessageToJavaObject[message];
        If[envelope === $Failed, Return[$Failed]];

        result = invoke[endPoint, envelope, options];
      ];
      
      result
    ], KeepObjects->{call}
  ]
  

(* Used for mapping parameters definitions with their values 
 * on methods that use a SOAPCall. 
 *)

mapParameters[paramDesc_List, parameters_List] := 
  Module[{},
    If[Length[paramDesc] == Length[parameters],
      MapThread[
        If[MatchQ[#1, (_String | 
                      {{_String, _String}, {___}} | 
                      {_String, {___}})], 
          {GetQualifiedName[#1], GetAttributes[#1]}->#2, 
          Message[InvokeServiceOperation::"mapparam", #1, #2];
          Return[$Failed]
        ] &, 
        {paramDesc, parameters}
      ],
      Message[InvokeServiceOperation::"mapparam", 
              paramDesc, 
              parameters];
      Return[$Failed];
    ]
  ]

ServiceOptionQ[x___] := MatchQ[x, _Symbol\[Rule]_]

ServiceOptionQ[x_List]:= MatchQ[x,{___?ServiceOptionQ}]

(* InstallServiceOperation *)

InstallServiceOperation[ServicesImpl["Java"],
                        exprName_Symbol, 
                        endPoint_String,
                        args_List,
                        header_List,
                        options___?OptionQ] :=
    JavaBlock[
      Module[{argNames, argTypes, argSymbols, argPats, 
              headerNames, headerTypes, headerSymbols},
        InstallJava[];

        {allowShortContext} = {"AllowShortContext"} /. 
          canonicalOptions[Flatten[{options}]]  /. Options[InstallServiceOperation];

        If[TrueQ[allowShortContext],
          context = Context[exprName];
          If[!MemberQ[$ContextPath, context], AppendTo[$ContextPath, context]];
        ];
          
        argNames = (Context[exprName] <> "Private`" <> NormalizeSymbolName[GetLocalName[#]] & /@  args);
        argTypes = GetType[#, MapNamespaces[#, {}]] & /@ args;
        argSymbols = Unique[ToExpression[#]] & /@ argNames;
        argPats = MapThread[Pattern[#1, GetPattern[#2]] &, {argSymbols, argTypes}];

        headerNames = (Context[exprName] <> "Private`" <> NormalizeSymbolName[GetLocalName[#]] & /@ header);   
        headerTypes = GetType[#, MapNamespaces[#, {}]] & /@ header;
        headerSymbols = Unique[ToExpression[#]] & /@ headerNames;
        headerPats = MapThread[Pattern[#1, GetPattern[#2]] &, {headerSymbols, headerTypes}];  
                           
        With[{exprName = exprName, argSymbols = argSymbols, 
              argDesc = args, argPats = Sequence @@ argPats, 
              headerSymbols = headerSymbols, headerDesc = header, 
              headerPats = Sequence @@ headerPats,
              o = options, optSymbol = ToExpression[Context[exprName] <> "Private`" <> "opts"], 
              optPat = Pattern @@ {ToExpression[Context[exprName] <> "Private`" <> "opts"], ___?ServiceOptionQ}},

          exprName[envelope:XMLObject["Document"][{___}, 
                     XMLElement[
                       ("Envelope" | 
                        {_String, "Envelope"}), 
                       {___}, 
                       {__XMLElement}], 
                     {___}], 
                   optPat] := 
            Module[{opts},
              opts = Union[Flatten[{optSymbol}], Flatten[{o}]];
                                                        
              InvokeServiceOperation[ServicesImpl["Java"], endPoint, envelope, opts]                  
            ];

          exprName[argPats, headerPats, optPat] := 
            Module[{envelope, result, opts, params, headerParams},
                                                        
              opts = Union[Flatten[{optSymbol}], Flatten[{o}]];
              params = mapParameters[argDesc, argSymbols];
              headerParams = mapParameters[headerDesc, headerSymbols];

              envelope = ToServiceRequest[ServicesImpl["Java"], params, headerParams, opts];
              If[envelope === $Failed, Return[$Failed]];  

              result = InvokeServiceOperation[ServicesImpl["Java"], endPoint, envelope, opts];
              If[result === $Failed, Return[$Failed]];  
                 
              FromServiceResponse[ServicesImpl["Java"], result, opts]
                
            ];
                    
          ToServiceRequest[exprName, argPats, headerPats, optPat] := 
            Module[{opts, params, headerParams},
              opts = Union[Flatten[{optSymbol}], Flatten[{o}]];
              params = mapParameters[argDesc, argSymbols];
              headerParams = mapParameters[headerDesc, headerSymbols];

              ToServiceRequest[ServicesImpl["Java"], params, headerParams, opts]
            ];
            
          InvokeServiceOperation[
                  exprName,
                  envelope:XMLObject["Document"][{___}, 
                     XMLElement[
                       ("Envelope" | 
                        {_String, "Envelope"}), 
                       {___}, 
                       {__XMLElement}], 
                     {___}], 
                   optPat] := 
            Module[{opts},
              opts = Union[Flatten[{optSymbol}], Flatten[{o}]];
                                                        
              InvokeServiceOperation[ServicesImpl["Java"], endPoint, envelope, opts]                  
            ];

          InvokeServiceOperation[exprName, argPats, headerPats, optPat] := 
            Module[{opts, params, headerParams, envelope}, 
              opts = Union[Flatten[{optSymbol}], Flatten[{o}]];
              params = mapParameters[argDesc, argSymbols];
              headerParams = mapParameters[headerDesc, headerSymbols];

              envelope = ToServiceRequest[ServicesImpl["Java"], params, headerParams, opts];
              If[envelope === $Failed, Return[$Failed]];  

              InvokeServiceOperation[ServicesImpl["Java"], endPoint, envelope, opts]
            ];
        ];
        
        $InstalledServices = Union[$InstalledServices, {exprName}];
        
        exprName
      ]
    ]
    
InstallService[ServicesImpl["Java"],
               wsdlURL_String, 
               options___?OptionQ] := 
  Module[{result},
    InstallJava[];
             
    If[TrueQ[$PrintPerformanceNumbers], Print["Installing Service: ", Date[]]];

    result = MapWSDL[ServicesImpl["Java"], wsdlURL, options];
          
    If[TrueQ[$PrintPerformanceNumbers], Print["Finished Installing Service: ", Date[]]];
    
    result
  ]

InstallService[ServicesImpl["Java"],
               wsdlURL_String,
               context_String,
               options___?OptionQ] :=
  Module[{result},
    InstallJava[];
        
    If[TrueQ[$PrintPerformanceNumbers], Print["Installing Service: ", Date[]]];
          
    result = MapWSDL[ServicesImpl["Java"], wsdlURL, context, options];      
    
    If[TrueQ[$PrintPerformanceNumbers], Print["Finished Installing Service: ", Date[]]];
        
    result
  ]

UninstallServiceOperation[ServicesImpl["Java"], operation_Symbol] :=
  (
    GetOperationPalette[operation] =.;
        
    $InstalledServices = Select[$InstalledServices, # =!= operation &];
    
    Remove[operation];
    
    (* 
    Clear[ToServiceRequest[operation, ___]];
    Clear[InvokeServiceOperation[operation, ___]];
    *)
  )
  
UninstallService[ServicesImpl["Java"], context_String] :=
  (     
    UninstallServiceOperation /@ Intersection[$InstalledServices, ToExpression /@ Names[context]];
  )

OperationPalette[ServicesImpl["Java"], operation_Symbol] := 
  Module[{notebook},        
    notebook = GetOperationPalette[operation];
    If[notebook === Null,
      Return["Palette cannot be found."];
     ];
    nb = NotebookPut[notebook];
  ]

  (* reportError is used to present Java errors in a prettier and less intimidating manner. *)    
  reportShortError[e_JLink`Package`manualException, 
              isPost : (True | False) : False] := 
    reportShortError[First[e], isPost]

  reportShortError[e_JLink`Package`autoException, 
              isPost : (True | False) : False]:= 
    reportShortError[First[e], isPost]    
    
  reportShortError[e_String, isPost : (True | False)] := 
   Block[{$MessagePrePrint, str = e, pos1, pos2},
     (*Sometimes get newline or 0x00 chars at end of message.Remove 
       them, and also the terminating period if there is one.*)
     str = FromCharacterCode[ToCharacterCode[e] /. {a__, junk___?(#<=13&)} :> {a}];
     If[StringTake[str, -1] === ".", str = StringDrop[str, -1]];

     If[StringMatchQ[str, "AxisFault*"],
       pos1 = StringPosition[str, "faultString: "];
       pos2 = StringPosition[str, "faultActor: "];
       str = StringTake[str, {Last[First[pos1]], First[First[pos2]] - 3}],
       If[StringMatchQ[str, "*\tat "],
         pos1 = StringPosition[str, "\tat "];
         str = StringTake[str, {1, First[First[pos1]] - 3}]
       ];
     ];
     Message[InvokeServiceOperation::"native", str]
   ]
   
   reportError[e_JLink`Package`manualException, 
                   isPost:(True | False):False] := 
     reportError[First[e], isPost]
     
   reportError[e_JLink`Package`autoException, 
                   isPost:(True | False):False] :=                   
     reportError[First[e], isPost]
     
   reportError[e_String, isPost:(True | False)] :=
    Block[{$MessagePrePrint, str = e},
      (* Sometimes get newline or 0x00 chars at end of message. Remove them, and also the terminating
         period if there is one.
      *)
      str = FromCharacterCode[ToCharacterCode[e] /. {a__, junk___?(#<=13&)} :> {a}];
      If[StringTake[str, -1] === ".", str = StringDrop[str, -1]];
      If[isPost, 
        Message[Java::pexcptn, str],
      (* else *)
        Message[Java::excptn, str]
      ]
    ]
  
End[]