(* :Author: Chris Williamson *)

Begin["`Package`"];

Normalize::usage =
	"Normalize is a utility function that uses WSDL definitions to normalize web service
	input and output.";

End[];

Begin["`Normalize`Private`"]

setName[_, name:{_String, _String}, _, _] := name;

setName[namespace_String, name_String, x_, defaultForm_] := 
  Module[{ns = ""},
    If[QualifiedFormQ[x, defaultForm], ns = namespace];    
    {ns, name}
  ];

setName[_, name_String,_,_] := {"", name};

setType[
  ({(_String | {_String, _String}), 
      {___,{_String?SchemaInstanceNamespaceQ, "type"}->_,___}}->_), 
  {_String, _String},
  "encoded",
  {"http://schemas.xmlsoap.org/soap/encoding/"}] := Null;

setType[
  XMLElement[
    (_String | {_String, _String}), 			
		{___,{_String?SchemaInstanceNamespaceQ, "type"}->_,___}, 
		{___}],
  {_String, _String},
  "encoded",
  {"http://schemas.xmlsoap.org/soap/encoding/"}] := Null;

setType[
  _, 
  t:{_String, _String},
  "encoded",
  {"http://schemas.xmlsoap.org/soap/encoding/"}] := 
    {"http://www.w3.org/2001/XMLSchema-instance","type"}->t;

setType[___] := Null;

setArrayType[
  XMLElement[
    (_String | {_String, _String}), 			
		{___,{"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->_,___}, 
		{___}], 
  _, 
  _List,
  "encoded",
  {"http://schemas.xmlsoap.org/soap/encoding/"}] := Null;

setArrayType[
  ({(_String | {_String, _String}), 
      {___,{"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->_,___}}->_), 
  _, 
  _List,
  "encoded",
  {"http://schemas.xmlsoap.org/soap/encoding/"}] := Null;

setArrayType[
  _, 
  t:{_String, _String},
  namespaces_List,
  "encoded",
  {"http://schemas.xmlsoap.org/soap/encoding/"}] := 
  Module[{schemaNamespaces, typeSchema, arrayType, type, defaultForm},

    {type, typeSchema, schemaNamespaces, defaultForm} = GetTypeContent[t];     
    
    If[(arrayType = GetArrayType[typeSchema]) =!= Null,
      {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->GetType[arrayType, schemaNamespaces],
      Null
    ]    
  ];

setArrayType[
  _, 
  type:XMLElement[{_?SchemaNamespaceQ, ("complexType" | "simpleType")}, {___}, {___}],
  namespaces_List,
  "encoded",
  {"http://schemas.xmlsoap.org/soap/encoding/"}] :=
  Module[{arrayType},
    If[(arrayType = GetArrayType[type]) =!= Null,
      {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->GetType[arrayType, namespaces],
      Null
    ]        
  ];

setArrayType[___] := Null;

(* Normalize Referenced Schema Element *)
Normalize[ServicesImpl["Java"],
          m_,
          element:XMLElement[
            {_?SchemaNamespaceQ, "element"}, 
            {___,{"", "ref"}->ref_String,___}, 
            {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{elmnt, e, nspaces, eSchema,defaultForm},

    elmnt = GetQName[ref, namespaces];
    {e, eSchema, nspaces, defaultForm} = GetElementContent[GetQName[ref, nspaces]];
    Normalize[
      ServicesImpl["Java"],
      m,
      eSchema,
      First[e],
      nspaces,
      True,
      encodingStyle, 
      encodingStyleURI
    ]
  ];

(* Normalize base type with string type *)
Normalize[ServicesImpl["Java"],
          (val:(True | False) | val_Integer | val_Real | val_List), 
          type_String,
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=

	Module[{t, newAttrs = {}, newName = {"", "item"}, typeAttr}, 

    t = GetType[type, namespaces];
	
    (* set type *)     
    If[(typeAttr = setType[val, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

 		{newName, newAttrs}->val
	];
  
(* Normalize base type with list type *)
Normalize[ServicesImpl["Java"],
          (val:(True | False) | val_Integer | val_Real | val_List), 
          type:({_String, _String} | {}),
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newAttrs = {}, newName = {"", "item"}, typeAttr},
      
    (* set type *)     
    If[(typeAttr = setType[val, type, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

 		{newName, newAttrs}->val
  ];

(* Normalize base type with schema type *)
Normalize[ServicesImpl["Java"],
          (val:(True | False) | val_Integer | val_Real | val_List), 
          type:XMLElement[{_?SchemaNamespaceQ, ("complexType" | "simpleType")}, 
            {___,{"", "name"}->typeName_String,___}, {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newAttrs = {}, newName = {"", "item"}, typeAttr},
      
    (* set type *)     
    If[!StringQ[namespace], ns = "", ns = namespace];    
    If[(typeAttr = setType[val, {ns, typeName}, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      
    
 		{newName, newAttrs}->val
  ];

(* Normalize base type with schema type *)
Normalize[ServicesImpl["Java"],
          (val:(True | False) | val_Integer | val_Real | val_List), 
          element:XMLElement[
            {_?SchemaNamespaceQ, "element"}, 
            {___,{"", "name"}->name_String,___}, 
            {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{type, t, newAttrs = {}, newName, typeAttr, typeName, ns},
    
    newName = setName[namespace, name, element, defaultForm];
    
    type = GetElementType[element];
    If[type =!= String && type =!= Null,
      typeName = GetTypeName[type];
      If[!StringQ[namespace], ns = "", ns = namespace];
      t = {ns, typeName}, 
      t = GetType[type, namespaces]
    ];        
    (* set type *)     
    If[(typeAttr = setType[val, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      
    
    {newName, newAttrs}->val
  ];
 
(* Normalize base type with null type *)
Normalize[ServicesImpl["Java"],
          (val:(True | False) | val_Integer | val_Real | val_List), 
          Null,
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] := {{"", "item"}, {}}->val;

(* Normalize base type with string type *)
Normalize[ServicesImpl["Java"],
          val_String,
          type_String,
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=

	Module[{t, newAttrs = {}, newName = {"", "item"}, typeAttr}, 

    t = GetType[type, namespaces];
	
    (* set type *)     
    If[(typeAttr = setType[val, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

 		XMLElement[newName, newAttrs, {val}]
	];
  
(* Normalize base type with list type *)
Normalize[ServicesImpl["Java"],
          val_String,
          type:({_String, _String} | {}),
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newAttrs = {}, newName = {"", "item"}, typeAttr},
      
    (* set type *)     
    If[(typeAttr = setType[val, type, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

 		XMLElement[newName, newAttrs, {val}]
  ];

(* Normalize base type with schema type *)
Normalize[ServicesImpl["Java"],
          val_String,
          type:XMLElement[{_?SchemaNamespaceQ, ("complexType" | "simpleType")}, 
            {___,{"", "name"}->typeName,___}, {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newAttrs = {}, newName = {"", "item"}, typeAttr},
      
    (* set type *)     
    If[!StringQ[namespace], ns = ""];    
    If[(typeAttr = setType[val, {namespace, typeName}, encodingStyle, encodingStyleURI]) =!= Null,
      AppendTo[newAttrs, typeAttr]
    ];      
    
 		XMLElement[newName, newAttrs, {val}]
  ];

Normalize[ServicesImpl["Java"],
          val_String,
          element:XMLElement[
            {_?SchemaNamespaceQ, "element"}, 
            {___,{"", "name"}->name_String,___}, 
            {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{type, t, newAttrs = {}, newName, typeAttr, typeName, typeNS},

    newName = setName[namespace, name, element, defaultForm];
      
    type = GetElementType[element];
    If[type =!= String && type =!= Null,
      typeName = GetTypeName[type];
      If[!StringQ[namespace], typeNS = "", typeNS = namespace];
      t = {typeNS, typeName}, 
      t = GetType[type, namespaces]
    ];        
    (* set type *)     
    If[(typeAttr = setType[val, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      
    
    {newName, newAttrs}->val
];

 
(* Normalize base type with null type *)
Normalize[ServicesImpl["Java"],
          val_String,
          Null,
          namespace_,
          namespaces_,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] := XMLElement[{"", "item"}, {}, {val}];  
  
(* Normalize Rule with string type *)
Normalize[ServicesImpl["Java"],
          m:((name_String | 
             {(name_String | name:{ns_String, _String}), 
              attrs:{___}})->val_), 
          type_String,
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{t, newName, newAttrs = {}, typeAttr, arrayTypeAttr},
          
    If[Head[Unevaluated[attrs]] === List, newAttrs = attrs];

    t = GetType[type, namespaces];
    
    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

    (* set type *)   
    If[(typeAttr = setType[m, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, t, namespaces, encodingStyle, encodingStyleURI]) =!= Null,
      AppendTo[newAttrs, arrayTypeAttr]
    ];
        
 		{newName, newAttrs}->val
  ];

(* Normalize Rule with list type *)
Normalize[ServicesImpl["Java"],
          m:((name_String | 
             {(name_String | name:{ns_String, _String}), 
              attrs:{___}})->val_), 
          type:({_String, _String} | {}),
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newName, newAttrs = {}, typeAttr, arrayTypeAttr},
      
    If[Head[Unevaluated[attrs]] === List, newAttrs = attrs];

    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

    (* set type *)     
    If[(typeAttr = setType[m, type, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, type, namespaces, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, arrayTypeAttr]
    ];

 		{newName, newAttrs}->val
  ];

(* Normalize Rule with schema type *)
Normalize[ServicesImpl["Java"],
          m:((name_String | 
             {(name_String | name:{_String, _String}), 
              attrs:{___}})->val_), 
          type:XMLElement[
            {_?SchemaNamespaceQ, ("complexType" | "simpleType")}, 
            {___,{"", "name"}->typeName,___}, 
            {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newName, newAttrs = {}, typeAttr, arrayTypeAttr, ns},
      
    If[Head[Unevaluated[attrs]] === List, newAttrs = attrs];

    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

    (* set type *)     
    If[!StringQ[namespace], ns = "", ns = namespace];    
    If[(typeAttr = setType[m, {ns, typeName}, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, type, namespaces, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, arrayTypeAttr]
    ];

 		{newName, newAttrs}->val
  ];
 
(* Normalize Rule with schema element *)
Normalize[ServicesImpl["Java"],
          m:((name_String | 
             {(name_String | name:{_String, _String}), 
              attrs:{___}})->val_), 
          element:XMLElement[
            {_?SchemaNamespaceQ, "element"}, 
            {___}, 
            {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{type, t, newName, newAttrs = {}, typeAttr, arrayTypeAttr},
      
    If[Head[Unevaluated[attrs]] === List, newAttrs = attrs];
    
    (* set name *)
    newName = setName[namespace, name, element, defaultForm];
    
    type = GetElementType[element];
    If[!StringQ[type] && type =!= Null,
      typeName = GetTypeName[type];
      If[!StringQ[namespace], ns = "", ns = namespace];
      t = {ns, typeName}, 
      t = GetType[type, namespaces, namespace];
      type = t;
    ];
    
    (* set type *)     
    If[(typeAttr = setType[m, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      
    
    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, type, namespaces, encodingStyle, encodingStyleURI]) =!= Null, AppendTo[newAttrs, arrayTypeAttr]];

    (* set null *)     
    If[val === Null && GetElementNillable[element] === "true", 
      AppendTo[newAttrs, {"http://www.w3.org/2001/XMLSchema-instance","nil"}->"true"];
    ];
    {newName, newAttrs}->val
  ];

(* Normalize Rule with null type *)
Normalize[ServicesImpl["Java"],
          m:((name_String | 
             {(name_String | name:{ns_String, _String}), 
              attrs:{___}})->val_), 
          Null,
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newName, newAttrs = {}},
      
    If[Head[Unevaluated[attrs]] === List, newAttrs = attrs];

    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

 		{newName, newAttrs}->val
  ];
               
(* Normalize XMLElement with String type *)
Normalize[ServicesImpl["Java"],
          m:XMLElement[
            (name_String | name:{_String, _String}), 			
						attrs:{___}, 
						val:{___}], 
          type_String,
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{t, newName, newAttrs = attrs, typeAttr, arrayTypeAttr},
    
    t = GetType[type, namespaces];
    
    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

    (* set type *)     
    If[(typeAttr = setType[m, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, t, namespaces, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, arrayTypeAttr]
    ];

 		XMLElement[newName, newAttrs, val]
  ]

(* Normalize XMLElement with list type *)
Normalize[ServicesImpl["Java"],
          m:XMLElement[(name_String | name:{_String, _String}), 			
												attrs:{___}, 
												val:{___}], 
          type:({_String, _String} | {}),
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newName, newAttrs = attrs, typeAttr, arrayTypeAttr},
      
    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

    (* set type *)     
    If[(typeAttr = setType[m, type, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, type, namespaces, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, arrayTypeAttr]
    ];

 		XMLElement[newName, newAttrs, val]
  ];

(* Normalize XMLElement with schema type *)
Normalize[ServicesImpl["Java"],
          m:XMLElement[(name_String | name:{_String, _String}), 			
												attrs:{___}, 
												val:{___}], 
          type:XMLElement[{_?SchemaNamespaceQ, ("complexType" | "simpleType")}, 
            {___,{"", "name"}->typeName_String,___}, {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newName, newAttrs = attrs, typeAttr, arrayTypeAttr, ns},
      
    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

    (* set type *)     
    If[!StringQ[namespace], ns = "", ns = namespace];    
    If[(typeAttr = setType[m, {ns, typeName}, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      

    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, type, namespaces, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, arrayTypeAttr]
    ];

 		XMLElement[newName, newAttrs, val]
  ];

(* Normalize XMLElement with schema type *)
Normalize[ServicesImpl["Java"],
          m:XMLElement[(name_String | name:{_String, _String}),       
                        attrs:{___}, 
                        val:{___}], 
          element:XMLElement[
            {_?SchemaNamespaceQ, "element"}, 
            {___}, 
            {___}],
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newName, newAttrs = attrs, typeAttr, arrayTypeAttr, ns, type, t, typeName},
      
    (* set name *)
    newName = setName[namespace, name, element, defaultForm];

    type = GetElementType[element];
    If[!StringQ[type] && type =!= Null,
      typeName = GetTypeName[type];
      If[!StringQ[namespace], ns = "", ns = namespace];
      t = {ns, typeName}, 
      t = GetType[type, namespaces, namespace];
      type = t;
    ];        
    (* set type *)     
    If[(typeAttr = setType[m, t, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, typeAttr]
    ];      
    
    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[m, type, namespaces, encodingStyle, encodingStyleURI]) =!= Null, 
      AppendTo[newAttrs, arrayTypeAttr]
    ];

    (* set null *)     
    If[val === Null && GetElementNillable[element] === "true", 
      AppendTo[newAttrs, {"http://www.w3.org/2001/XMLSchema-instance","nil"}->"true"];
    ];

    XMLElement[newName, newAttrs, val]
  ];


(* Normalize XMLElement with null type *)
Normalize[ServicesImpl["Java"],
          m:XMLElement[(name_String | name:{_String, _String}), 			
												attrs:{___}, 
												val:{___}, ___], 
          Null,
          namespace_,
          namespaces_List,
          defaultForm_,
          encodingStyle_,
          encodingStyleURI_] :=
  Module[{newName, newAttrs = attrs},
     
    (* set name *)
    newName = setName[namespace, name, Null, defaultForm];

 		XMLElement[newName, newAttrs, val]
  ];

Normalize[_ServicesImpl, x___] := x;
      
End[];