(* :Author: Chris Williamson *)

Begin["`Package`"];

GeneratePalette::usage = 
  "GeneratePalette is a utility function that generates a OperationPalette palette.";

GetOperationPalette::usage = 
  "GetOperationPalette is a utility function that is used to retrieve a OperationPalette.";

AddMethodToPalette::usage = 
  "AddMethodToPalette is a utility function that is used to add meta data to the palette 
  definitions";

End[];

Begin["`OperationPalette`Private`"]

types = {};

methods = {};

baseTypeInformation["http://www.wolfram.com/XML/", "Expression"] :=
  "Expression is a Mathematica Expression.";

baseTypeInformation["http://www.w3.org/1998/Math/MathML", ("math"|"math.type")] :=
  "Math is a mathematical expression.  This is represented by Symbolic XML.";

baseTypeInformation[namespace_?BaseTypeNamespaceQ, name_String] :=
  Switch[name,
		"int", "int is derived from long by setting the value of maxInclusive to be 2147483647 and minInclusive to be -2147483648. The base type of int is long.  This is best represented by an Integer in Mathematica.",
 		"integer", "integer is derived from decimal by fixing the value of fractionDigits to be 0. This results in the standard mathematical concept of the integer numbers. The value space of integer is the infinite set {...,-2,-1,0,1,2,...}. The base type of integer is decimal.  This is best represented by an Integer in Mathematica.",
 		"double", "The double datatype corresponds to IEEE double-precision 64-bit floating point type [IEEE 754-1985]. The basic value space of double consists of the values m  2^e, where m is an integer whose absolute value is less than 2^53, and e is an integer between -1075 and 970, inclusive. In addition to the basic value space described above, the value space of double also contains the following special values: positive and negative zero, positive and negative infinity and not-a-number. The order-relation on double is: x < y iff y - x is positive. Positive zero is greater than negative zero. Not-a-number equals itself and is greater than all double values including positive infinity.  This is best represented by a Real in Mathematica.",
 		"float", " float corresponds to the IEEE single-precision 32-bit floating point type [IEEE 754-1985]. The basic value space of float consists of the values m  2^e, where m is an integer whose absolute value is less than 2^24, and e is an integer between -149 and 104, inclusive. In addition to the basic value space described above, the value space of float also contains the following special values: positive and negative zero, positive and negative infinity and not-a-number. The order-relation on float is: x < y iff y - x is positive. Positive zero is greater than negative zero. Not-a-number equals itself and is greater than all float values including positive infinity. This is best represented by a Real in Mathematica.",
 		"string", "The string datatype represents character strings in XML. The value space of string is the set of finite-length sequences of characters (as defined in XML 1.0 (Second Edition)) that match the Char production from XML 1.0 (Second Edition). A character is an atomic unit of communication; it is not further specified except to note that every character has a corresponding Universal Character Set code point, which is an integer.  This is best represented by a String in Mathematica.",
 		"boolean", "boolean has the value space required to support the mathematical concept of binary-valued logic: {true, false}. This is best represented using True and False in Mathematica",
 		"short", "short is derived from int by setting the value of maxInclusive to be 32767 and minInclusive to be -32768. The base type of short is int.  This is best represented by an Integer in Mathematica",
 		"long", " long is derived from integer by setting the value of maxInclusive to be 9223372036854775807 and minInclusive to be -9223372036854775808.  The base type of long is integer.  This is best represented by an Integer in Mathematica.",
 		"decimal", "decimal represents arbitrary precision decimal numbers. The value space of decimal is the set of the values i  10^-n, where i and n are integers such that n >= 0. The order-relation on decimal is: x < y iff y - x is positive.  The value space of types derived from decimal with a value for totalDigits of p is the set of values i  10^-n, where n and i are integers such that p >= n >= 0 and the number of significant decimal digits in i is less than or equal to p. The value space of types derived from decimal with a value for fractionDigits of s is the set of values i  10^-n, where i and n are integers such that 0 <= n <= s. This is best represented by a Real in Mathematica.",
 		"byte", "byte is derived from short by setting the value of maxInclusive to be 127 and minInclusive to be -128. The base type of byte is short.  This is best represented by an Integer in Mathematica.",
 		"nonPositiveInteger", "nonPositiveInteger is derived from integer by setting the value of maxInclusive to be 0. This results in the standard mathematical concept of the non-positive integers. The value space of nonPositiveInteger is the infinite set {...,-2,-1,0}. The base type of nonPositiveInteger is integer.  This is best represented by an Integer in Mathematica.",
 		"negativeInteger", "negativeInteger is derived from nonPositiveInteger by setting the value of maxInclusive to be -1. This results in the standard mathematical concept of the negative integers. The value space of negativeInteger is the infinite set {...,-2,-1}. The base type of negativeInteger is nonPositiveInteger.  This is best represented by an Integer in Mathematica.",
 		"nonNegativeInteger", "nonNegativeInteger is derived from integer by setting the value of minInclusive to be 0. This results in the standard mathematical concept of the non-negative integers. The value space of nonNegativeInteger is the infinite set {0,1,2,...}. The base type of nonNegativeInteger is integer.  This is bestrepresented by an Integer in Mathematica.",
 		"unsignedLong", "unsignedLong is derived from nonNegativeInteger by setting the value of maxInclusive to be 18446744073709551615. The base type of unsignedLong is nonNegativeInteger.  This is best represented by anInteger in Mathematica.",
 		"unsignedInt", "unsignedInt is derived from unsignedLong by setting the value of maxInclusive to be 4294967295. The base type of unsignedInt is unsignedLong.  This is best represented by an Integer in Mathematica.",
 		"unsignedShort", "unsignedShort is derived from unsignedInt by setting the value of maxInclusive to be 65535. The base type of unsignedShort is unsignedInt.  This is best represented by an Integer in Mathematica.",
 		"unsignedByte", "unsignedByte is derived from unsignedShort by setting the value of maxInclusive to be 255. The base type of unsignedByte is unsignedShort.  This is best represented by an Integer in Mathematica.",
 		"positiveInteger", "positiveInteger is derived from nonNegativeInteger by setting the value of minInclusive to be 1. This results in the standard mathematical concept of the positive integer numbers. The value space of positiveInteger is the infinite set {1,2,...}. The base type of positiveInteger is nonNegativeInteger. This is best represented by an Integer in Mathematica.",
		"duration", "duration represents a duration of time. The value space of duration is a six-dimensional space where the coordinates designate the Gregorian year, month, day, hour, minute, and second components defined in  5.5.3.2 of [ISO 8601], respectively. These components are ordered in their significance by their order of appearance i.e. as year, month, day, hour, minute, and second.  This is best represented by a String in Mathematica.",
 		"dateTime", "dateTime represents a specific instant of time. The value space of dateTime is the space of Combinations of date and time of day values as defined in  5.4 of [ISO 8601].  This is best represented by a String in Mathematica.", 
 		"time", "time represents an instant of time that recurs every day. The value space of time is the space of time of day values as defined in  5.3 of [ISO 8601]. Specifically, it is a set of zero-duration daily time instances.  This is best represented by a String in Mathematica.",
 		"date", "date represents a calendar date. The value space of date is the set of Gregorian calendar dates as defined in  5.2.1 of [ISO 8601]. Specifically, it is a set of one-day long, non-periodic instances e.g. lexical 1999-10-26 to represent the calendar date 1999-10-26, independent of how many hours this day has.  This is best represented by a String in Mathematica.",
 		"gYearMonth", "gYearMonth represents a specific gregorian month in a specific gregorian year. The value space of gYearMonth is the set of Gregorian calendar months as defined in  5.2.1 of [ISO 8601]. Specifically, it is a set of one-month long, non-periodic instances e.g. 1999-10 to represent the whole month of 1999-10, independent of how many days this month has. This is best represented by a String in Mathematica.",
 		"gYear", "gYear represents a gregorian calendar year. The value space of gYear is the set of Gregorian calendar years as defined in  5.2.1 of [ISO 8601]. Specifically, it is a set of one-year long, non-periodic instances e.g. lexical 1999 to represent the whole year 1999, independent of how many months and days this year has.  This is bestrepresented by a String in Mathematica.",
 		"gMonthDay", "gMonthDay is a gregorian date that recurs, specifically a day of the year such as the third of May. Arbitrary recurring dates are not supported by this datatype. The value space of gMonthDay is the set of calendar dates, as defined in  3 of [ISO 8601]. Specifically, it is a set of one-day long, annually periodic instances. This is best represented by a String in Mathematica.",
 		"gDay", "gDay is a gregorian day that recurs, specifically a day of the month such as the 5th of the month. Arbitrary recurring days are not supported by this datatype. The value space of gDay is the space of a set of calendar dates as defined in  3 of [ISO 8601]. Specifically, it is a set of one-day long, monthly periodic instances.  This is best represented by a String in Mathematica.",
 		"gMonth", "gMonth is a gregorian month that recurs every year. The value space of gMonth is the space of a set of calendar months as defined in  3 of [ISO 8601]. Specifically, it is a set of one-month long, yearly periodic instances.  This is best represented by a String in Mathematica.",
 		"base64Binary", "base64Binary represents Base64-encoded arbitrary binary data. The value space of base64Binary is the set of finite-length sequences of binary octets. For base64Binary data the entire binary stream is encoded using the Base64 Content-Transfer-Encoding defined in Section 6.8 of [RFC 2045].  This is best represented bya String in Mathematica.",
 		"hexBinary", "hexBinary represents arbitrary hex-encoded binary data. The value space of hexBinary is the set of finite-length sequences of binary octets. This is best represented by a String in Mathematica.",
 		"anyURI", "anyURI represents a Uniform Resource Identifier Reference (URI). An anyURI value can be absolute or relative, and may have an optional fragment identifier (i.e., it may be a URI Reference). This type should be used to specify the intention that the value fulfills the role of a URI as defined by [RFC 2396], as amended by [RFC 2732]. This is best represented by a String in Mathematica.",
 		"QName", "QName represents XML qualified names. The value space of QName is the set of tuples {namespace name, local part}, where namespace name is an anyURI and local part is an NCName. The lexical space of QName is the set of strings that match the QName production of [Namespaces in XML].  This is best represented by a List of two Strings in Mathematica.",
 		"normalizedString", "normalizedString represents white space normalized strings. The value space of normalizedString is the set of strings that do not contain the carriage return (#xD), line feed (#xA) nor tab (#x9) characters. The lexical space of normalizedString is the set of strings that do not contain the carriage return (#xD) nor tab (#x9) characters. The base type of normalizedString is string.  This is best represented by a String in Mathematica.",
 		"token", "token represents tokenized strings. The value space of token is the set of strings that do not contain the line feed (#xA) nor tab (#x9) characters, that have no leading or trailing spaces (#x20) and that have no internal sequences of two or more spaces. The lexical space of token is the set of strings that do not contain the line feed (#xA) nor tab (#x9) characters, that have no leading or trailing spaces (#x20) and that have no internal sequences of two or more spaces. The base type of token is normalizedString.  This is best represented by a String in Mathematica.",
 		"language", "language represents natural language identifiers as defined by [RFC 1766]. The value space of language is the set of all strings that are valid language identifiers as defined in the language identification section of [XML 1.0 (Second Edition)]. The lexical space of language is the set of all strings that are valid language identifiers as defined in the language identification section of XML 1.0 (Second Edition). The base type of language is token.  Thisis best represented by a String in Mathematica.",
 		"Name", "Name represents XML Names. The value space of Name is the set of all strings which match the Name production of XML 1.0 (Second Edition). The lexical space of Name is the set of all strings which match the Name production of XML 1.0 (Second Edition). The base type of Name is token.  This is best represented by a String in Mathematica.", 
 		"NCName", "NCName represents XML \"non-colonized\" Names. The value space of NCName is the set of all strings which match the NCName production of [Namespaces in XML]. The lexical space of NCName is the set of all strings which match the NCName production of [Namespaces in XML]. The base type of NCName is Name.  This is best represented by a String in Mathematica.",
 		"NMTOKEN", "NMTOKEN represents the NMTOKEN attribute type from XML 1.0 (Second Edition). The value space of NMTOKEN is the set of tokens that match the Nmtoken production in XML 1.0 (Second Edition). The lexical space of NMTOKEN is the set of strings that match the Nmtoken production in XML 1.0 (Second Edition). The base type of NMTOKEN is token. This is best represented by a String in Mathematica.",
 		"NMTOKENS", "NMTOKENS represents the NMTOKENS attribute type from XML 1.0 (Second Edition). The value space of NMTOKENS is the set of finite, non-zero-length sequences of NMTOKENs. The lexical space of NMTOKENS is the set of white space separated lists of tokens, of which each token is in the lexical space of NMTOKEN. The itemType of NMTOKENS is NMTOKEN.  This is best represented by a String in Mathematica",
 		"ID", "ID represents the ID attribute type from XML 1.0 (Second Edition). The value space of ID is the set of all strings that match the NCName production in [Namespaces in XML]. The lexical space of ID is the set of all strings that match the NCName production in [Namespaces in XML]. The base type of ID is NCName. This is bestrepresented by a String in Mathematica.", 
 		"IDREF", "IDREF represents the IDREF attribute type from XML 1.0 (Second Edition). The value space of IDREF is the set of all strings that match the NCName production in [Namespaces in XML]. The lexical space of IDREF is the set of strings that match the NCName production in [Namespaces in XML]. The base type of IDREF is NCName.  This is bestrepresented by a String in Mathematica.",
 		"IDREFS", "IDREFS represents the IDREFS attribute type from XML 1.0 (Second Edition). The value space of IDREFS is the set of finite, non-zero-length sequences of IDREFs. The lexical space of IDREFS is the set of white space separated lists of tokens, of which each token is in the lexical space of IDREF. The itemType of IDREFS is IDREF.  This is best represented by a String in Mathematica.",
 		"ENTITY","ENTITY represents the ENTITY attribute type from XML 1.0 (Second Edition). The value space of ENTITY is the set of all strings that match the NCName production in [Namespaces in XML] and have been declared as an unparsed entity in a document type definition. The lexical space of ENTITY is the set of all strings that match the NCName production in [Namespaces in XML]. The base type of ENTITY is NCName.  This is best represented by a Stringin Mathematica.",
 		"ENTITIES", "ENTITIES represents the ENTITIES attribute type from XML 1.0 (Second Edition). The value space of ENTITIES is the set of finite, non-zero-length sequences of ENTITYs that have been declared as unparsed entities in a document type definition. The lexical space of ENTITIES is the set of white space separated lists of tokens, of which each token is in the lexical space of ENTITY. The itemType of ENTITIES is ENTITY.  This is best represented by a String in Mathematica.",
 		"Array", "SOAP arrays are defined as having a type of \"SOAP-ENC:Array\" or a type derived there from. Arrays are represented as element values, with no specific constraint on the name of the containing element (just as values generally do not constrain the name of their containing element).  Arrays can contain elements which themselves can be of any type, including nested arrays. New types formed by restrictions of SOAP-ENC:Array can also be created to represent, for example, arrays limited to integers or arrays of some user-defined enumeration.  The representation of the value of an array is an ordered sequence of elements constituting the items of the array. Within an array value, element names are not significant for distinguishing accessors. Elements may have any name. In practice, elements will frequently be named so that their declaration in a schema suggests or determines their type. As with compound types generally, if the value of an item in the array is a single-reference value, the item contains its value. Otherwise, the item references its value via an \"href\" attribute.  This is best represented by a Listin Mathematica.",
 		"base64", "The recommended representation of an opaque array of bytes is the 'base64' encoding defined in XML Schemas, which uses the base64 encoding algorithm defined in RFC2045.  However, the line length restrictions that normally apply to base64 data in MIME do not apply in SOAP. A \"SOAP-ENC:base64\" subtype is supplied for use with SOAP.  This is best represented by a String in Mathematica.",
 		_, "The type {" <> namespace <> "}" <> name <> " is not recognized in this namespace."
 	]

baseTypeInformation[___] := "No Information available"  

generateTypeInformation[type:{"http://www.w3.org/1998/Math/MathML", ("math"|"math.type")}] :=
  Cell[
    CellGroupData[
      {Cell[
        baseTypeInformation["http://www.w3.org/1998/Math/MathML", "math.type"], 
        CellTags->{"{http://www.w3.org/1998/Math/MathML}math.type Info"}]}, 
      Closed]];

generateTypeInformation[type:{"http://www.wolfram.com/XML/", "Expression"}] :=
  Cell[
    CellGroupData[
      {Cell[
        baseTypeInformation["http://www.wolfram.com/XML/", "Expression"], 
        CellTags->{"{http://www.wolfram.com/XML/}Expression Info"}]}, 
      Closed]];
  
generateTypeInformation[type:{namespace_?BaseTypeNamespaceQ, name_String}] :=
  Cell[
    CellGroupData[
      {Cell[
        baseTypeInformation[namespace, name], 
        CellTags->{"{" <> namespace <> "}" <> name <> " Info"}]}, 
      Closed]];

generateTypeInformation[type:{namespace_String, name_String}] :=
  Module[{t, nspaces, typeSchema,defaultForm},
    {t, typeSchema, nspaces, defaultForm} = GetTypeContent[type];      	  
 	  generateTypeInformation[{namespace, name, typeSchema, nspaces}]    
  ]

generateTypeInformation[
  {namespace_String, name_String, typeSchema_XMLElement, nspaces_List}] :=
  Module[{arrayType, list = {}, u, params, t, enumerations, defaultForm},
      
    arrayType = GetArrayType[typeSchema];
    If[arrayType =!= Null, arrayType = GetType[arrayType, nspaces]];
    If[MatchQ[arrayType, {_String, _String}],  		
  		If[StringMatchQ[Last[arrayType], "*[*"], 
        arrayType = 
          {First[arrayType], 
           StringTake[
             Last[arrayType], 
             {1, First[First[StringPosition[Last[arrayType], "["]]]-1}]};
   		];
    ];
    
    enumerations = GetEnumerations[typeSchema];
   	
		params = GetElements[typeSchema];		
    If[MatchQ[arrayType, {_String, _String}],
      list = 
        {Cell[
          TextData[
            {ButtonBox["{" <> First[arrayType] <> "}" <> Last[arrayType],
             ButtonData->"{" <> First[arrayType] <> "}" <> Last[arrayType] <> " Info",
             ButtonStyle->"Hyperlink"]}],
          CellTags->{"{" <> namespace <> "}" <> name <> " Info"}]},
      If[enumerations =!= {},
        list = 
          {Cell[
             BoxData[StyleBox["Enumeration Values", FontWeight->"Bold"]], 
             CellTags->{"{" <> namespace <> "}" <> name <> " Info"}]};
        AppendTo[
          list, 
          Cell[
            BoxData[RowBox[{"- ", 
				    ButtonBox[#, 
				    ButtonData->ToBoxes[#],
            ButtonStyle->"Paste"]}]]]] & /@ enumerations,
        (* If not an expression *)
   		  list = 
          {Cell[
            BoxData["* name -> type"], 
            CellTags->{"{" <> namespace <> "}" <> name <> " Info"}]};
          (u = "* ";
            Block[{schemaNspaces = nspaces, schema = #, typeNS = namespace, t, e},
              schemaNspaces = MapNamespaces[#, nspaces];
              If[GetElementRef[#] =!= Null,
                {e, schema, schemaNspaces,defaultForm} = GetElementContent[GetQName[GetElementRef[#], schemaNspaces]];
                 typeNS = First[e];
               ];         
               u = u <> "{" <> typeNS <> "}" <> GetElementName[schema] <> "->";
               t = GetElementType[schema];
               If[StringQ[t], t = GetType[t, schemaNspaces, typeNS]];
               If[MatchQ[
                 t, 
                 XMLElement[
                   {_?SchemaNamespaceQ, ("simpleType" | "complexType")}, 
                   {___}, 
                   {___}]],
                 t = {typeNS, GetTypeName[t]};
               ];
               If[MatchQ[t, {_String, _String}],
                 AppendTo[
                   list, 
                   Cell[
                     TextData[{
                        RowBox[{
                          u, 
                          ButtonBox["{" <> First[t] <> "}" <> Last[t], 
                          ButtonData->"{" <> First[t] <> "}" <> Last[t] <> " Info",
                          ButtonStyle->"Hyperlink"]}]}]]];           
               ];
            ];
          ) & /@ params;
      ];
		];
		Cell[CellGroupData[list, Closed]]
  ]
  
stringBaseTypeQ[name_String] := 
  StringMatchQ[name, "string"] ||
  StringMatchQ[name, "duration"] || 
  StringMatchQ[name, "dateTime"] || 
  StringMatchQ[name, "time"] || 
  StringMatchQ[name, "date"] || 
  StringMatchQ[name, "gYearMonth"] || 
  StringMatchQ[name, "gYear"] || 
  StringMatchQ[name, "gMonthDay"] || 
  StringMatchQ[name, "gDay"] || 
  StringMatchQ[name, "gMonth"] || 
  StringMatchQ[name, "base64Binary"] || 
  StringMatchQ[name, "hexBinary"] || 
  StringMatchQ[name, "anyURI"] || 
  StringMatchQ[name, "QName"] || 
  StringMatchQ[name, "normalizedString"] || 
  StringMatchQ[name, "token"] || 
  StringMatchQ[name, "language"] || 
  StringMatchQ[name, "Name"] || 
  StringMatchQ[name, "NCName"] || 
  StringMatchQ[name, "NMTOKEN"] || 
  StringMatchQ[name, "NMTOKENS"] || 
  StringMatchQ[name, "IX"] || 
  StringMatchQ[name, "IDREF"] || 
  StringMatchQ[name, "IDREFS"] || 
  StringMatchQ[name, "ENTITY"] || 
  StringMatchQ[name, "ENTITIES"] || 
  StringMatchQ[name, "base64"]

addMethod[operation_Symbol, method:{_List, _List, _, _}] := 
  Module[{m, cases},
    m = getMethods[operation];
    m = Union[m, {method}];
    cases = Cases[methods, {operation, _}];
    If[Length[cases] > 0, 
      methods = ReplaceAll[methods, {operation, _List}->{operation, m}],
      methods = Append[methods, {operation, m}]
    ];
  ]

getMethods[operation_Symbol] := 
  Module[{cases},
    cases = Cases[methods, {operation, _List}];
    If[Length[cases] > 0, 
      Return[cases[[1,2]]],
      Return[{}]
    ]
  ]

sameTypeQ[{ns1_String, name1_String, ___}, {ns2_String, name2_String, ___}] := 
  (ns1 === ns2 && name1 === name2);
  
sameTypeQ[___] := False;
  
addType[operation_Symbol, type:{___}] := 
  Module[{t, cases},
    t = getTypes[operation];
    t = Union[t, {type}, SameTest->sameTypeQ];
    cases = Cases[types, {operation, _}];
    If[Length[cases] > 0, 
      types = ReplaceAll[types, {operation, _List}->{operation, t}],
      types = Append[types, {operation, t}]
    ];
  ]

getTypes[operation_Symbol] := 
  Module[{cases},
    cases = Cases[types, {operation, _List}];
    If[Length[cases] > 0, 
      Return[cases[[1,2]]],
      Return[{}]
    ]
  ]

setType[
  ({(_String | {_String, _String}), 
     {___,{_String?SchemaInstanceNamespaceQ, "type"}->_,___}}->_), 
  {_String, _String}] := Null;

setType[
  XMLElement[
    (_String | {_String, _String}), 			
		{___, {_String?SchemaInstanceNamespaceQ, "type"}->_, ___}, 
 		{___}],
  {_String, _String}] := Null;

setType[_, t:{_String, _String}] := {"http://www.w3.org/2001/XMLSchema-instance","type"}->t;

setType[___] := Null;

setArrayType[
  XMLElement[
    (_String | {_String, _String}), 			
		{___,{"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->_,___}, 
		{___}], 
  _, 
  _List] := Null;

setArrayType[
  ({(_String | {_String, _String}), 
     {___,{"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->_,___}}->_), 
  _, 
  _List] := Null;

setArrayType[
  _, 
  {_String, _String},
  _List] := 
  Module[{nspaces, typeSchema, arrayType, type,defaultForm},

    {type, typeSchema, nspaces,defaultForm} = GetTypeContent[type];     

    If[(arrayType = GetArrayType[typeSchema]) =!= Null,
      {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->
        GetType[arrayType, nspaces],
      Null
    ]    
  ];

setArrayType[
  _, 
  type:XMLElement[
    {_?SchemaNamespaceQ, ("complexType" | "simpleType")}, 
    {___}, 
    {___}],
  namespaces_List] :=
  Module[{arrayType},
    If[(arrayType = GetArrayType[type]) =!= Null,
      {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->GetType[arrayType, namespaces],
      Null
    ]        
  ];

setArrayType[___] := Null;

newTypeTemplate[
  operation_Symbol, 
  name:(_String | {_String, _String}), 
	type:{_String, _String},
	options___?OptionQ] := 
 	Module[{prefix, longForm, returnName, arrayTypeAttr, typeAttr, value, 
 	        attrs = {}, m, opts = canonicalOptions[Flatten[{options}]]},      

    {longForm, returnName} = {"LongForm", "ReturnName"} /. opts  /. {"LongForm"->False, "ReturnName"->True};
    opts = DeleteCases[opts, "ReturnName"->_];
    		       
 		If[TrueQ[longForm],

      If[(typeAttr = setType[m, type]) =!= Null, AppendTo[attrs, typeAttr]];      

      (* set array type *)    
      If[(arrayTypeAttr = setArrayType[m, type, namespaces]) =!= Null, 
        AppendTo[attrs, arrayTypeAttr]
      ];

 		];
 		
    value = getTypeTemplate[operation, type, opts];
    
    If[longForm === True && returnName =!= False, 
      prefix = {name, attrs}, 
      If[returnName =!= False, 
        If[ListQ[name], prefix = Last[name], prefix = name], 
        prefix = Null
      ];
    ];
    
    (* Return value *)
    If[MatchQ[prefix, Null], 
      value,
      prefix->value
    ]
	]

newTypeTemplate[
  operation_Symbol, 
  name:(_String | {_String, _String}), 
	typeInfo:{
	  namespace_String, 
	  typeName_String,
	  type:XMLElement[
      {_?SchemaNamespaceQ, ("simpleType" | "complexType")}, 
      {___,{"","name"}->typeName_String,___}, 
      {___}],
	  namespaces_List
	},
	options___?OptionQ] := 
 	Module[{prefix, longForm, returnName, arrayTypeAttr, typeAttr, value, 
 	        attrs = {}, m, opts = canonicalOptions[Flatten[{options}]]},      

    {longForm, returnName} = {"LongForm", "ReturnName"} /. opts  /. {"LongForm"->False, "ReturnName"->True};
    opts = DeleteCases[opts, "ReturnName"->_];
        		       
 		If[TrueQ[longForm],

      If[(typeAttr = setType[m, {namespace, GetTypeName[type]}]) =!= Null, 
        AppendTo[attrs, typeAttr]
      ];      

      (* set array type *)    
      If[(arrayTypeAttr = setArrayType[m, type, namespaces]) =!= Null, 
        AppendTo[attrs, arrayTypeAttr]
      ];
 		];

    value = getTypeTemplate[operation, typeInfo, opts];

    If[longForm === True && returnName =!= False, 
      prefix = {name, attrs}, 
      If[returnName =!= False, 
        If[ListQ[name], prefix = Last[name], prefix = name], 
        prefix = Null
      ];
    ];
    
    (* Return value *)
    If[MatchQ[prefix, Null], 
      value,
      prefix->value
    ]
	]

newTypeTemplate[
  operation_Symbol,
  type:{"http://www.w3.org/1998/Math/MathML", ("math" | "math.type")},
  options___?OptionQ] :=
  ( 
    addType[operation, type];
    Global`\[Placeholder]
  );

newTypeTemplate[
  operation_Symbol,
  type:{"http://www.wolfram.com/XML/", "Expression"},
  options___?OptionQ] :=
  ( 
    addType[operation, type];
    Global`\[Placeholder]
  );

newTypeTemplate[
  operation_Symbol,
	type:{_?BaseTypeNamespaceQ, typeName_?BaseTypeQ},
  ___?OptionQ] := 
  Module[{value = Global`\[Placeholder]},
    If[stringBaseTypeQ[typeName],
      value = "\[Placeholder]";
    ];
    addType[operation, type];
    value
  ];

newTypeTemplate[
  operation_Symbol,
	type:{namespace_String, typeName_String},
  options___?OptionQ] := 
  Module[{t, nspaces, typeSchema,defaultForm,
          opts = canonicalOptions[Flatten[{options}]]},
    {t, typeSchema, nspaces,defaultForm} = GetTypeContent[type];
    opts = DeleteCases[opts, "ReturnName"->_];     
    getTypeTemplate[
      operation, 
      {namespace, typeName, typeSchema, nspaces}, 
      opts]
  ];
	
newTypeTemplate[
  operation_Symbol,
	 typeInfo:{
	   namespace_String, 
	   typeName_String,
	   typeSchema:XMLElement[
       {_?SchemaNamespaceQ, ("simpleType" | "complexType")}, 
       {___,{"","name"}->typeName_String,___}, 
       {___}], 
	   nspaces_List
	 },
  options___?OptionQ] := 
	Module[
    {value = Global`\[Placeholder], arrayType, params, 
     restriction, restrictionType, opts = canonicalOptions[Flatten[{options}]]},

    restriction = GetRestrictionBase[typeSchema];
    restrictionType = GetType[restriction, nspaces];
    If[MatchQ[restrictionType, {_String, _String}], 
      addType[operation, typeInfo];
      Return[newTypeTemplate[operation, restrictionType, opts]];
    ];
    
    arrayType = GetArrayType[typeSchema];
    If[arrayType =!= Null, arrayType = GetType[arrayType, nspaces]];

    If[MatchQ[arrayType, {_String, _String}],  		
  		If[StringMatchQ[Last[arrayType], "*[*"], 
        arrayType = {
          First[arrayType], 
          StringTake[
            Last[arrayType], 
            {1, First[First[StringPosition[Last[arrayType], "["]]]-1}]};
   		];
    ];
 		params = GetElements[typeSchema];	
    If[arrayType === Null,
      If[IsArray[typeSchema],
        AppendTo[opts, "ReturnName"->False];
      ];
      value =
        (Block[{schemaNspaces = nspaces, schema = #, typeNS = namespace, e, t},
           schemaNspaces = MapNamespaces[#, nspaces];
           If[GetElementRef[#] =!= Null,
             {e, schema, schemaNspaces, defaultForm} = GetElementContent[GetQName[GetElementRef[#], schemaNspaces]];
             typeNS = First[e];
           ];          
           t = GetElementType[schema];
           If[StringQ[t], t = GetType[t, schemaNspaces, typeNS]];
           If[MatchQ[
                t, 
                XMLElement[
                  {_?SchemaNamespaceQ, ("simpleType" | "complexType")}, 
                  {___}, 
                  {___}]],
             newTypeTemplate[
               operation, 
               {typeNS, GetElementName[schema]}, 
		           {typeNS, GetTypeName[t], t, schemaNspaces}, 
	             opts],
             If[MatchQ[t, {_String, _String}],
  	          newTypeTemplate[
	              operation, 
		             {typeNS, GetElementName[schema]}, 
		             t,
	               opts],
	             \[Placeholder]
	           ]
	         ]]) & /@ params,
      AppendTo[opts, "ReturnName"->False];      
      value = {getTypeTemplate[operation, arrayType, opts]}
    ];
		
    addType[operation, typeInfo];
    If[arrayType =!= Null, addType[operation, arrayType]];
      
	  value
	]
	
	newTypeTemplate[
    _Symbol,
	  ___,
	  ___?OptionQ] := Global`\[Placeholder]
	
	generateTypeCells[operation_Symbol, items_List] := 
		Module[{list}, 
			list = 
			  Cell[CellGroupData[
 				{
          Cell[BoxData[
                RowBox[
                  {
	 	                ButtonBox[Part[#,2],
			      	    		 ButtonData->ToBoxes[
			        	   	 		 getTypeTemplate[operation, #, LongForm->False]],
	   		 	  		       ButtonStyle->"Paste"], 
	   		 	  		    ButtonBox["Long Form",
											 ButtonData->ToBoxes[
											   getTypeTemplate[operation, #, LongForm->True]],
	   		 	  		       ButtonStyle->"Paste"]
	   		 	  		  }]
		      ], "Subsubsection"],
		      generateTypeInformation[#]
				}, Closed]] & /@ items;

			Cell[CellGroupData[
			  {
			    Cell[items[[1,1]], "Subsection"], 
			    CellGroupData[list, Closed]
			  }, Closed]]
		];
		
	generateParameterCells[
    operation_,
	  m:((name_String | {(name_String | {_String, name_String}), {___}}))] :=
	  Module[{type, template = Global`\[Placeholder], 
            attrTemplate = "\[Placeholder]", typeName = "Not defined"},
	    type = GetType[m, {}];
	    If[MatchQ[type, {_String, _String}],
	      template = 
          ToBoxes[InputForm[getTypeTemplate[operation, type, LongForm->False]]];
	      attrTemplate = ToBoxes[getTypeTemplate[operation, type, LongForm->True]];
	      typeName = "{" <> type[[1]] <> "}" <> type[[2]];
      ];
      Cell[CellGroupData[{
	      Cell[TextData[{RowBox[
	        {
	          ButtonBox[name,
			 		  ButtonData->template,
 		          ButtonStyle->"Paste"],
	          StyleBox["->"], 
	          ButtonBox[typeName,
	            ButtonData->typeName <> " Info",
              ButtonStyle->"Hyperlink"]
	        }]}],"Subsubsection"],
        Cell["Template:", "Text", FontWeight->"Bold"],
        Cell[BoxData[template], "Text"],
        Cell[BoxData[ButtonBox["Long Form",
       	  ButtonData->attrTemplate,
 		      ButtonStyle->"Paste"]], "Text"]
        }, Closed]]      
	  ]
	
	generateMethodCells[
    operation_Symbol, 
    {params_List, headerParams_List, returnType_, doc_}] :=
	  Module[{names = ToString[operation]<>"[ ", p = {}, h = {}, 
            r = StyleBox["Not Defined"], headerTemplatesWithAttrs,
            template = Global`\[Placeholder], templates, 
            headerTemplates, templatesWithAttrs, 
	          d = "Documentation was not provided with WSDL.  However, descriptions of the parameters are provided below."},
         
      If[doc =!= Null && StringQ[doc], 
        d = doc
      ];

      If[returnType =!= Null && MatchQ[returnType, {_String, _String}], 
        r = ButtonBox["{"<>returnType[[1]]<>"}"<>returnType[[2]], 
              ButtonData->"{"<>returnType[[1]]<>"}"<>returnType[[2]] <> " Info",
				      ButtonStyle->"Hyperlink"];
				template = 
          ToBoxes[InputForm[getTypeTemplate[operation, returnType, LongForm->False]]];
      ];
      
      (names = 
        names <> GetLocalName[#] <> 
          ToString[GetPattern[GetType[#, MapNamespaces[#, {}]]]] <> ", ") & /@ params;
      (names = 
        names <> GetLocalName[#] <> 
          ToString[GetPattern[GetType[#, MapNamespaces[#, {}]]]] <> ", ") & /@ headerParams;
      If[Length[params] > 0 || Length[headerParams] > 0,
        names = StringDrop[names, -2];
      ];
      names = names <> " ]";

      p = generateParameterCells[operation, #] & /@ params;
			h	=	generateParameterCells[operation, #] & /@ headerParams;
			
			templates = 
        getTypeTemplate[operation, GetType[#, {}], LongForm->False] & /@ params;
			headerTemplates = 
        getTypeTemplate[operation, GetType[#, {}], LongForm->False] & /@ headerParams;
      templatesWithAttrs = 
        getTypeTemplate[operation, GetType[#, {}], LongForm->True] & /@ params;
			headerTemplatesWithAttrs = 
        getTypeTemplate[operation, GetType[#, {}], LongForm->True] & /@ headerParams;

      opTemplate = 
        RowBox[{
          ToString[operation], "[", 
          RowBox[
            Drop[
              Drop[
                First[ToBoxes[Join[templates, headerTemplates]]], -1], 1]], 
          "]"
        }];
      
      opWithAttrTemplate = 
			   RowBox[{
			     ToString[operation], "[", 
			     RowBox[
             Drop[
               Drop[
                 First[
                   ToBoxes[
                     Join[templatesWithAttrs, headerTemplatesWithAttrs]]], -1], 1]], 
           "]"
        }];
              
      Cell[CellGroupData[{
        Cell[names, "Subsection"],
        Cell[BoxData[RowBox[
	        {
	          ButtonBox["Paste",
  			 		  ButtonData->opTemplate,
 		          ButtonStyle->"Paste"],
 		        ButtonBox["Long Form",
						  ButtonData->opWithAttrTemplate,
 		          ButtonStyle->"Paste"]
	        }]],"text"],          
        Cell[d, "text"],
        Cell[CellGroupData[p]],
        Cell[CellGroupData[h]],
        Cell[CellGroupData[{
          Cell[
            TextData[{RowBox[{StyleBox["Return Type: "], r }]}], 
            "Subsubsection", 
            FontWeight->"Bold"],
          Cell["Template:", "Text", FontWeight->"Bold"],
          Cell[BoxData[template], "Text"]
        }, Closed]]
      }, Closed]]
	  ]
	
  AddMethodToPalette[
    ServicesImpl["Java"], 
    operation_Symbol, 
    doc_, 
    params_List, 
    headerParams_List, 
    returnType_] :=
      addMethod[operation, {params, headerParams, returnType, doc}];
  
	GeneratePalette[ServicesImpl["Java"], operation_Symbol, wsdl_String] :=
    Module[{notebook, methods = {}, types = {}},       
			
			methods = getMethods[operation];
    
      getTypeTemplate[o_Symbol, t:{___}, opts___?OptionQ] := 
        getTypeTemplate[o, t, opts] = newTypeTemplate[o, t, opts];
      
			m = generateMethodCells[operation, #] & /@ methods;
			
			types = getTypes[operation];
      
			types = Sort[types, OrderedQ[{Part[#1,1], Part[#2,1]}] &];
			types = Split[types,StringMatchQ[Part[#1,1], Part[#2,1]] &];
			types = Union /@ types;
			
			t = generateTypeCells[operation, #] & /@ types;
	    
      Clear[getTypeTemplate];
      
      notebook = 
        Notebook[
          {
            Cell[operation, "Subtitle"],
            Cell[TextData[{
              ButtonBox[wsdl, 
                ButtonData:>{URL[wsdl],None}, 
                ButtonStyle->"Hyperlink"]}], "text"], 
            Cell["Operations", "Section"],
            Cell[CellGroupData[m]],
            Cell[CellGroupData[{
              Cell["Types", "Section"],
              Cell[CellGroupData[t]]
            }, Closed]]
          },
        FrontEndVersion->"4.2 for Microsoft Windows",
				ScreenRectangle->{{0, 1024}, {0, 723}},
			  ClosingAutoSave->False,
				Editable->False,
				WindowSize->{500, 700},
 				WindowMargins->{{Automatic, 1}, {Automatic, 0}},
				WindowFrame->"Normal",
				WindowElements->"VerticalScrollBar",
				WindowFrameElements->{"CloseBox", "ZoomBox", "ResizeArea"},
				WindowClickSelect->False,
				WindowTitle->"Operation Palette",
				ShowCellBracket->False,
				Active->True,
				Background->GrayLevel[1],
				StyleDefinitions -> Notebook[{
				  Cell["\<\
				          Prototypes for the BasicCalculations Palette. This abbreviated \
					style sheet is used to reduce the size of the notebook as a whole while still \
					allowing the styles to be easily modifed. Remember that all other styles will \
					be adopted from the default style sheet.          \
					\>", "Text"],
						  
				  Cell[StyleData[All, "Working"],
			    ShowCellBracket->False,
			    PageWidth->WindowWidth,
			    ScriptMinSize->9],
							  
				  Cell[StyleData[All, "Presentation"],
 						   PageWidth->WindowWidth,
							 ScriptMinSize->12,
						   FontSize->16],
							  
					Cell[StyleData[All, "Condensed"],
							 PageWidth->WindowWidth,
						   CellBracketOptions->{"Margins"->{1, 1},
						   "Widths"->{0, 5}},
						   ScriptMinSize->8,
						   FontSize->11],
								  
					Cell[StyleData[All, "Printout"],
							    PageWidth->PaperWidth,
							    ScriptMinSize->5,
							    FontSize->10,
							    PrivateFontOptions->{"FontType"->"Outline"}],
								  
					Cell[CellGroupData[{
							  
						Cell[StyleData["Section"],
      		  	CellDingbat->None,
    				  ShowGroupOpenCloseIcon->True,
				    	CellMargins->{{19, Inherited}, {3, 18}},
						  CellGroupingRules->{"SectionGrouping", 30},
						  PageBreakBelow->False,
						  CounterIncrements->"Section",
						  CounterAssignments->{{"Subsection", 0}, {"Subsubsection", 0}},
						  FontFamily->"Helvetica",
						  FontSize->14,
						  FontWeight->"Bold"],
							  
					  Cell[StyleData["Section", "Presentation"],
						  CellMargins->{{24, Inherited}, {Inherited, 20}},
						  LineSpacing->{1, 0},
						  FontSize->18],
								  
					  Cell[StyleData["Section", "Printout"],
						  CellMargins->{{14, Inherited}, {Inherited, 20}},
						  FontSize->14],
								  
					  Cell[StyleData["Section", "Condensed"],
						  CellMargins->{{4, Inherited}, {2, 2}},
						  FontSize->14]
						}, Closed]],
								  
					Cell[CellGroupData[{
								  
					  Cell[StyleData["Subsection"],
      		  	CellDingbat->None,
    				  ShowGroupOpenCloseIcon->True,
					    CellMargins->{{25, Inherited}, {Inherited, 8}},
					    CellGroupingRules->{"SectionGrouping", 50},
					    PageBreakBelow->False,
					    CounterIncrements->"Subsection",
					    FontFamily->"Times",
					    FontSize->12,
					    FontWeight->"Bold"],
							  
					  Cell[StyleData["Subsection", "Presentation"],
					    CellMargins->{{24, Inherited}, {Inherited, 15}},
					    LineSpacing->{1, 0},
					    FontSize->18],
								  
					  Cell[StyleData["Subsection", "Printout"],
					    CellMargins->{{14, Inherited}, {Inherited, 15}},
					    FontSize->12],
								  
					  Cell[StyleData["Subsection", "Condensed"],
					    CellMargins->{{4, Inherited}, {2, 2}},
					    FontSize->12]
					  }, Closed]],

					Cell[CellGroupData[{
								  
					  Cell[StyleData["Subsubsection"],
      		  	ShowCellBracket->False,
      		  	CellDingbat->None,
    				  ShowGroupOpenCloseIcon->True,
					    CellMargins->{{30, Inherited}, {Inherited, 8}},
					    CellGroupingRules->{"SectionGrouping", 70},
					    PageBreakBelow->False,
					    CounterIncrements->"Subsubsection",
					    FontFamily->"Times",
					    FontSize->12,
					    FontWeight->"Bold"],
							  
					  Cell[StyleData["Subsubsection", "Presentation"],
					    CellMargins->{{24, Inherited}, {Inherited, 15}},
					    LineSpacing->{1, 0},
					    FontSize->18],
								  
					  Cell[StyleData["Subsubsection", "Printout"],
					    CellMargins->{{14, Inherited}, {Inherited, 15}},
					    FontSize->10],
								  
					  Cell[StyleData["Subsubsection", "Condensed"],
					    CellMargins->{{4, Inherited}, {2, 2}},
					    FontSize->10]
					  }, Closed]],
								  
					Cell[CellGroupData[{
								  
					  Cell[StyleData["Text"],
					    CellMargins->{{35, Inherited}, {0, Inherited}},
					    FontFamily->"Times"],
								  
					  Cell[StyleData["Text", "Presentation"],
					    CellMargins->{{24, Inherited}, {10, Inherited}},
					    LineSpacing->{1, 0}],
								  
					  Cell[StyleData["Text", "Printout"],
					    CellMargins->{{14, Inherited}, {0, Inherited}},
					    FontSize->10],
								  
					  Cell[StyleData["Text", "Condensed"],
					    CellMargins->{{4, Inherited}, {0, 2}},
					    FontSize->10]
					  }, Closed]],
								  
					  Cell[CellGroupData[{
								  
					  Cell[StyleData["Input"],
					    CellMargins->{{35, Inherited}, {Inherited, Inherited}},
					    CellGroupingRules->"InputGrouping",
					    CellHorizontalScrolling->True,
					    PageBreakWithin->False,
					    GroupPageBreakWithin->False,
					    CellLabelMargins->{{23, Inherited}, {Inherited, Inherited}},
					    DefaultFormatType->DefaultInputFormatType,
					    FormatType->InputForm,
					    ShowStringCharacters->True,
					    NumberMarks->True,
					    FontWeight->"Bold"],
								  
					  Cell[StyleData["Input", "Presentation"],
					    CellMargins->{{45, Inherited}, {20, Inherited}},
					    LineSpacing->{1, 0}],
								  
					  Cell[StyleData["Input", "Printout"],
					    CellMargins->{{30, Inherited}, {10, Inherited}},
					    FontSize->10],
								  
					  Cell[StyleData["Input", "Condensed"],
					    CellMargins->{{4, Inherited}, {2, 2}},
					    FontSize->10]
					  }, Closed]]
					}]                  
        ];
                  
      notebook
    ]
  
GetOperationPalette[_] := Null

End[]
