(* :Author: Chris Williamson *)

Begin["`Package`"];

Serialize::usage =
	"Serialize is a utility function used to serialize a web service request into a format
	desired by the web service.";

NamespaceId::usage = 
  "NamespaceId is a utiltity symbol that is used to keep track of the last generic namespace
	id used while serializing a request message.";

End[];
	
Begin["`Serialize`Private`"]

InvokeServiceOperation::"srlz" = "Could not serialize: `1`"

(* Serialize and Deserialize *) 

base64 = Null;

base64Encode[str_String] := (
  If[base64 === Null, base64 = JavaNew["org.apache.axis.encoding.Base64"]];
  JavaObjectToExpression[base64@encode[MakeJavaObject[str]@getBytes[]]]
);

(* Used for mapping attributes to an element. If the value of the attribute is not a 
   String, Integer, or Real, the attribute will not be set.  Also the type
   attribute will not be set, as it is set earlier in Serialize.
 *)

namespaceQ = {};

getNamespacePrefix[ns_String, namespaces_List] := 
  Module[{nsMappings, prefixes, prefix},
    If[ns === "", Return[""]];
    nsMappings = Union[namespaces, namespaceQ];
    prefixes = Select[nsMappings, MatchQ[#, {_?XMLNamespaceQ, val_String}->ns] &];
    If[Length[prefixes] > 0, 
      prefix = prefixes[[1, 1, 2]],
      prefix = "ns"<> ToString[NamespaceId];
      NamespaceId++;
      AppendTo[namespaceQ, {"http://www.w3.org/2001/xmlns/", prefix}->ns];
    ];
    prefix
  ]

(* QName Attributes in a List - only in Rule syntax *)

mapAttribute[
  {ns_String, name_String}->{attrNS_String, attrName_String}, 
  attributes:{___}, 
  namespaces_List] :=
 	Append[
    attributes, 
    {getNamespacePrefix[ns, namespaces], name}->
      getNamespacePrefix[attrNS, namespaces]<>":"<>attrName]

mapAttribute[
  name_String->{attrNS_String, attrName_String}, 
  attributes:{___},
  namespaces_List] :=
	Append[attributes, name->getNamespacePrefix[attrNS, namespaces]<>":"<>attrName]

(* QName Attributes in a String *)

mapAttribute[
  {ns_String, name_String}->attrVal_String?WrappedStringQ,
  attributes:{___},
  namespaces_List] :=
  Module[{val = attrVal, position, namespace},
		position = Part[StringPosition[val, "}"], 1, 2];
		namespace = StringTake[val, {2, position - 1}];
		val = StringTake[val, {position+1, StringLength[val]}];
   	Append[
      attributes, 
      {getNamespacePrefix[ns, namespaces], name}->
        getNamespacePrefix[namespace,namespaces]<>":"<>val]
	]

mapAttribute[
  name_String->attrVal_String?WrappedStringQ,
  attributes:{___},
  namespaces_List] :=
  Module[{val = attrVal, position, namespace},
		position = Part[StringPosition[val, "}"], 1, 2];
		namespace = StringTake[val, {2, position - 1}];
		val = StringTake[val, {position+1, StringLength[val]}];
   	Append[attributes, {ns, name}->getNamespacePrefix[namespace, namespaces]<>":"<>val]
	]

(* Regualar Attributes *)

mapAttribute[
  {ns_String, name_String}->val_, 
  attributes:{___},
  namespaces_List] :=
	Module[{attrVal},
		Switch[ 
			val, 
			True, attrVal = "true",
			False, attrVal = "false",
			_String, attrVal = val,
			_Integer, attrVal = ToString[val],
			_Real, attrVal = ToString[NumberForm[val, 16, NumberFormat -> (If[#3 == "", #1,SequenceForm[#1, "e", #3]] &)]],
			_, attrVal = Null;
		];	
		If[attrVal =!= Null,
			Append[attributes, {getNamespacePrefix[ns, namespaces], name}->attrVal],
			attributes
	  ]
	]

mapAttribute[
  name_String->val_, 
  attributes:{___}, 
  namespaces_List] :=
	Module[{attrVal},
		Switch[ 
			val, 
			True, attrVal = "true",
			False, attrVal = "false",
			_String, attrVal = val,
			_Integer, attrVal = ToString[val],
			_Real, attrVal = ToString[NumberForm[val, 16, NumberFormat -> (If[#3 == "", #1,SequenceForm[#1, "e", #3]] &)]],
			_, attrVal = Null;
		];
		If[attrVal =!= Null,
			Append[attributes, name->attrVal],
			attributes
	  ]
	]

serializeChild[child_, nspaces_List, typeSchema_, typeNS_, typeNspaces_, typeDF_, options_] := 
  Module[{encodingStyle, encodingStyleURI, name, childNS = typeNS, defaultForm = typeDF,
          childSchema, childNspaces = typeNspaces, n, d, type = {}},
  
    {encodingStyle, encodingStyleURI} = 
      {"EncodingStyle", "EncodingStyleURI"} /. 
        canonicalOptions[Flatten[{options}]]  /. Options[InvokeServiceOperation];

    name = GetName[child];
    Switch[name,
      _String,
        childSchema = GetElementDef[name, typeSchema];
        childNspaces = MapNamespaces[childSchema, typeNspaces];
        childNS = typeNS;
        defaultForm = typeDF,
      {_String, _String},
        If[StringMatchQ[typeNS, First[name]],
          childSchema = GetElementDef[Last[name], typeSchema];
          childNspaces = MapNamespaces[childSchema, typeNspaces];
          childNS = typeNS;
          defaultForm = typeDF;
          If[childSchema == Null,
            {name, childSchema, childNspaces, defaultForm} = GetElementContent[name]
          ],
          {name, childSchema, childNspaces, defaultForm} = GetElementContent[name];
          childNS = First[name];
        ],
      _,
        If[IsArray[typeSchema],
          childSchema = GetElementDef[Null, typeSchema];
          childNspaces = MapNamespaces[childSchema, typeNspaces];
          defaultForm = typeDF;
          childNS = typeNS, 
          childSchema = Null;
          childNspaces = Null;
          childNS = {};
        ] 
    ];
    eType = GetElementType[childSchema];
    If[!StringQ[eType] && eType =!= Null,
      type = {childNS, GetTypeName[eType]},
      type = GetType[eType, childNspaces, childNS]
    ];
    n = Normalize[ServicesImpl["Java"], child, childSchema, childNS, childNspaces, defaultForm, encodingStyle, encodingStyleURI];
    d = Serialize[ServicesImpl["Java"], n, type, nspaces, options];
    d
  ]


setValue[
  ((_String | {(_String | {_String, _String}), 
     {___, 
      {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->arrayType_,
      ___}})->val_List), 
  type_, 
  nspaces_List, 
  options___?OptionQ] := 
  Module[{t = Null, n, namespace = "", encodingStyle, encodingStyleURI}, 
   
    {encodingStyle, encodingStyleURI} = 
      {"EncodingStyle", "EncodingStyleURI"} /. 
        canonicalOptions[Flatten[{options}]]  /. Options[InvokeServiceOperation];
   
    If[MatchQ[type, {_String, _String}], namespace  = First[type]];

		If[StringQ[arrayType] && !StringMatchQ[arrayType, "*[*"], 
      t = StringTake[arrayType, {1, First[First[StringPosition[arrayType, "["]]]-1}];
      t = GetType[t, nspaces];
 		];
 		 		
    If[MatchQ[arrayType, {_String, _String}],  		
  		If[StringMatchQ[Last[arrayType], "*[*"], 
        t = {First[arrayType], StringTake[Last[arrayType], {1, First[First[StringPosition[Last[arrayType], "["]]]-1}]};
   		];
    ];
    
    children = (
      (n = Normalize[ServicesImpl["Java"], #, t, namespace, nspaces, False, encodingStyle, encodingStyleURI];
       Serialize[ServicesImpl["Java"], n, t, nspaces, options])& /@ val)
  ]

setValue[
  ((_String | {(_String | {_String, _String}), 
    {___}})->val_), 
  type:{"http://www.w3.org/1998/Math/MathML", "math.type"}, 
  nspaces_List, 
  options___?OptionQ] := 
  Module[{children, t, typeSchema, schemaNamespaces, typeNS, defaultForm}, 
    
    {t, typeSchema, schemaNamespaces,defaultForm} = GetTypeContent[type];     
    If[MatchQ[type, {_String, _String}], typeNS = First[t]];
        
    children = (serializeChild[#, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, options] & /@ val[[3]])
  ]

setValue[
  ((_String | {(_String | {_String, _String}), 
     {___}})->val_), 
  type:{"http://www.wolfram.com/XML/", "Expression"}, 
  nspaces_List, 
  options___?OptionQ] := 
  Module[{expr, children, t, typeSchema, typeNS, schemaNamespaces,defaultForm}, 
    
    expr = XML`ToVerboseXML[XML`NotebookML`ExpressionToSymbolicExpressionML[val]];
    If[expr === Null, Return[$Failed]];
    
    {t, typeSchema, schemaNamespaces,defaultForm} = GetTypeContent[type];     
    If[MatchQ[type, {_String, _String}], typeNS = First[t]];

    (* Could just pass expr[[2]] into serialChild but the symbolic XML that is generated
       contains attributes that are not valid when exporting back to a string.  *)
    children = (serializeChild[#, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, options] & /@ expr[[2,3]])
  ]
  
setValue[
  param:((_String | {(_String | {namespace_String, _String}), 
     {___}})->val_),
  type_, 
  nspaces_List, 
  options___?OptionQ] := 
  Module[{typeNS = namespace, typeSchema, defaultForm,
          n, children, failed, schemaNamespaces = {}, t},
       
    {t, typeSchema, schemaNamespaces,defaultForm} = GetTypeContent[type];     
    If[MatchQ[t, {_String, _String}], typeNS = First[type]];
		Switch[
			val,
			_List, 
      	children = 
          (serializeChild[#, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, options] & /@ val);
		    failed = Cases[children, $Failed, Infinity];
			  If[Length[failed] > 0, Return[$Failed]],
			_Rule,
        children = serializeChild[val, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, options];
				If[children =!= $Failed, children = {children}],
			_XMLElement, 
        children = serializeChild[val, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, options];
				If[children =!= $Failed, children = {children}],
			_String, 
			  If[MatchQ[type, {_String?SchemaNamespaceQ, ("base64"|"base64Binary")}], 
			    children = {base64Encode[val]},
			    children = {val};
			  ],
			_Integer, children = {ToString[val]},
			_Real, children = {ToString[NumberForm[val, 16, NumberFormat -> (If[#3 == "", #1, SequenceForm[#1, "e", #3]] &)]]},
			True, children = {"true"},
			False, children = {"false"},
			Null, children = {},
			_, Message[InvokeServiceOperation::"srlz", val];Return[$Failed];
		];
		children
  ]

(* Serialize Rule *)

Serialize[
  ServicesImpl["Java"], 
  parameter:((name_String | {(name_String | {namespace_String, name_String}), 
     attrs:{___}})->val_), 
  (type:{_String, _String} | type:{}), 
  namespaces_List,
  options___?OptionQ] :=
	Module[{encodingStyle, encodingStyleURI, nspaces, nm, attributes = {}, children},
    
    {encodingStyle, encodingStyleURI} = {"EncodingStyle", "EncodingStyleURI"} /. 
      canonicalOptions[Flatten[{options}]]  /. Options[InvokeServiceOperation];
				 
		(* Get options and add namespaces *)
		nspaces = MapNamespaces[parameter, namespaces];
				
	  If[Head[Unevaluated[namespace]] === String,
	    nm = {getNamespacePrefix[namespace, nspaces], name},
	    nm = name
	  ];
	  		
	  (* Add attributes *)
		If[Head[Unevaluated[attrs]] === List, 
		  (attributes = mapAttribute[#, attributes, nspaces]) & /@ attrs;
		];
		
		(* Add namespaces required for attributes *)
    (attributes = mapAttribute[#, attributes, nspaces]) & /@ namespaceQ;
    nspaces = Union[nspaces, namespaceQ];
    namespaceQ = {};
	  
	  children = setValue[parameter, type, nspaces, options];
	  If[children === $Failed, Return[$Failed]];
	  
		XMLElement[nm, attributes, children]
	]

XMLElementQ[x_XMLElement] := True

XMLElementQ[x___] := (Message[InvokeServiceOperation::"srlz", x];False)

setValue[
  XMLElement[
    (_String | {_String, _String}), 			
		{___, {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->arrayType_,___}, 
		children:{___}], 
  type_, 
  nspaces_List, 
  options___?OptionQ] := 
  Module[{t = Null, c, namespace = "", typeSchema, typeNS, schemaNamespaces, failed, defaultForm},
 
    If[MatchQ[type, {_String, _String}], namespace  = First[type]];

		If[StringQ[arrayType] && !StringMatchQ[arrayType, "*[*"], 
      t = StringTake[arrayType, {1, First[First[StringPosition[arrayType, "["]]]-1}];
 		];
 		 		
    If[MatchQ[arrayType, {_String, _String}],  		
  		If[StringMatchQ[Last[arrayType], "*[*"], 
        t = {First[arrayType], StringTake[Last[arrayType], {1, First[First[StringPosition[Last[arrayType], "["]]]-1}]};
   		];
    ];
    
    {t, typeSchema, schemaNamespaces,defaultForm} = GetTypeContent[t];     
    If[MatchQ[type, {_String, _String}], typeNS = First[t]];

		c = (If[XMLElementQ[#], 
          serializeChild[#, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, options],
  				$Failed
  			 ] & /@ children),
		failed = Cases[c, $Failed, Infinity];
		If[Length[failed] > 0, $Failed, c]
  ]
  
setValue[m:XMLElement[(name_String | {ns_String, name_String}), 			
											attrs:{___}, 
											children:{___}],
         type_, 
         nspaces_List, 
         options___?OptionQ] := 
  Module[{encodingStyle, encodingStyleURI, typeNS = ns, schema, typeSchema, 
          n, failed, schemaNamespaces = {}, c = children, t,defaultForm},

    {encodingStyle, encodingStyleURI} = {"EncodingStyle", "EncodingStyleURI"} /. 
      canonicalOptions[Flatten[{options}]]  /. Options[InvokeServiceOperation];
    
    {t, typeSchema, schemaNamespaces,defaultForm} = GetTypeContent[type];     
    If[MatchQ[t, {_String, _String}], typeNS = First[type]];
  
		(* Process children and/or set value *)
		Switch[children,
		  {___String}, c = children,
		  {___XMLElement}, 
     		c = 
     		  (If[XMLElementQ[#],
            serializeChild[#, nspaces, typeSchema, typeNS, schemaNamespaces, defaultForm, options],
  			  	$Failed
    			 ] & /@ children);
				failed = Cases[c, $Failed, Infinity];
 				If[Length[failed] > 0, Return[$Failed]],
 			_, c = $Failed
		];
		c
  ]

(* Serialize XMLElement *)
Serialize[
  ServicesImpl["Java"], 
  m:XMLElement[
    (name_String | {ns_String, name_String}), 			
	  attrs:{___}, 
		{___}], 
  (type:{_String, _String} | type:{}), 
	namespaces_List,
	options___?OptionQ] :=
  
	Module[{nspaces, nm, attributes = {}, qname, jtype, node, refType = Null, c}, 
		
		(* Get options *)
		nspaces = MapNamespaces[m, namespaces];

    If[Head[Unevaluated[ns]] === String,
	    nm = {getNamespacePrefix[ns, nspaces], name},
	    nm = name
	  ];
	  
		(* Add attributes *)
		If[Head[Unevaluated[attrs]] === List, 
		  (attributes = mapAttribute[#, attributes, nspaces]) & /@ attrs;
		];
		
		(* Add namespaces required for attributes *)
    (attributes = mapAttribute[#, attributes, nspaces]) & /@ namespaceQ;
    nspaces = Union[nspaces, namespaceQ];
    namespaceQ = {};
	  
	  c = setValue[m, type, nspaces, options];
	  If[c === $Failed, Return[$Failed]];

		XMLElement[nm, attributes, c]
	]

Serialize[_ServicesImpl, x___] := (
	Message[InvokeServiceOperation::"srlz", {x}];
	$Failed 
);

	
End[]
