(* :Author: Chris Williamson *)

Begin["`Package`"];

BaseTypeNamespaceQ;
BaseTypeQ;
XMLNamespaceQ;
WrappedQNameQ;
PrefixedQNameQ;
GetQName;
GetType;
GetName;
GetQualifiedName;
GetLocalName;
GetAttribute;
GetAttributes;
GetValue;
GetSymbolicXML;
GetPattern;
MapNamespaces;
NormalizeSymbolName;
GetSchema;
GetTypeContent;
GetElementContent;
GetArrayType;
IsExpression;

GetSchemaDefinitions::usage =
  "GetSchemaDefinitions is a utility function that provides access to Schema definitions.";

End[];

Begin["`Utilities`Private`"]

GetSchemaDefinitions[___] := {};

(* Used to determine whether a namespace is in the basetype namespace *)

BaseTypeNamespaceQ[x_String] :=   
  StringMatchQ[x, "http://www.w3.org/1999/XMLSchema"] || 
  StringMatchQ[x, "http://www.w3.org/2000/XMLSchema"] || 
  StringMatchQ[x, "http://www.w3.org/2001/XMLSchema"  ] ||
  StringMatchQ[x, "http://schemas.xmlsoap.org/soap/encoding/"]
  
BaseTypeNamespaceQ[___] := False

(* Used to determine whether the local part of a type is a basetype *) 

BaseTypeQ[x_String] :=   
  StringMatchQ[x, "boolean"] || 
  StringMatchQ[x, "int"] || 
  StringMatchQ[x, "integer"] ||
  StringMatchQ[x, "double"] ||
  StringMatchQ[x, "float"] || 
  StringMatchQ[x, "string"] || 
  StringMatchQ[x, "short"] || 
  StringMatchQ[x, "long"] || 
  StringMatchQ[x, "decimal"] || 
  StringMatchQ[x, "byte"] || 
  StringMatchQ[x, "nonPositiveInteger"] || 
  StringMatchQ[x, "negativeInteger"] || 
  StringMatchQ[x, "nonNegativeInteger"] || 
  StringMatchQ[x, "unsignedLong"] || 
  StringMatchQ[x, "unsignedInt"] || 
  StringMatchQ[x, "unsignedShort"] || 
  StringMatchQ[x, "unsignedByte"] || 
  StringMatchQ[x, "positiveInteger"] || 
  StringMatchQ[x, "duration"] || 
  StringMatchQ[x, "dateTime"] || 
  StringMatchQ[x, "time"] || 
  StringMatchQ[x, "date"] || 
  StringMatchQ[x, "gYearMonth"] || 
  StringMatchQ[x, "gYear"] || 
  StringMatchQ[x, "gMonthDay"] || 
  StringMatchQ[x, "gDay"] || 
  StringMatchQ[x, "gMonth"] || 
  StringMatchQ[x, "base64Binary"] || 
  StringMatchQ[x, "hexBinary"] || 
  StringMatchQ[x, "anyURI"] || 
  StringMatchQ[x, "QName"] || 
  StringMatchQ[x, "normalizedString"] || 
  StringMatchQ[x, "token"] || 
  StringMatchQ[x, "language"] || 
  StringMatchQ[x, "Name"] || 
  StringMatchQ[x, "NCName"] || 
  StringMatchQ[x, "base64"] || 
  StringMatchQ[x, "anyType"]

BaseTypeQ[___] := False

(* Used to identify new namespaces *)
XMLNamespaceQ[x_String] := StringMatchQ[x, "http://www.w3.org/1999/xmlns/"] || 
         StringMatchQ[x, "http://www.w3.org/2000/xmlns/"] || 
         StringMatchQ[x, "http://www.w3.org/2001/xmlns/"] || 
         StringMatchQ[x, "xmlns"]

XMLNamespaceQ[___] := False

WrappedQNameQ[x_String] := StringMatchQ[x, "{*}*"]

WrappedQNameQ[___] := False

GetQName[t_String?WrappedQNameQ, namespaces_List] := 
  Module[{position, prefix, namespace, type},
    type = t;
    position = Part[StringPosition[type, "}"], 1, 2];
    namespace = StringTake[type, {2, position - 1}];
    type = StringTake[type, {position+1, StringLength[type]}];
    {namespace, type}
  ]

PrefixedQNameQ[x_String] := StringMatchQ[x, "*:*"]

PrefixedQNameQ[___] := False

GetQName[t_String?PrefixedQNameQ, namespaces_List] :=
  Module[{position, prefix, namespace, type},
    type = t;
    position = First[First[StringPosition[type, ":"]]];
    prefix = StringTake[type, {1, position - 1}];
    namespace = Select[namespaces, MatchQ[#, {_?XMLNamespaceQ, prefix}->_]&];
    If[MatchQ[namespace, {__Rule}], namespace = namespace[[1,2]], prefix = "None"];
    type = StringTake[type, {position+1, StringLength[type]}];
    If[StringMatchQ[prefix, "None"], {"", t}, {namespace, type}]
  ] 

(* 
 * QName for Strings without a prefix or namespace
 *) 
GetQName[t_String, namespaces_List] := {"", t}

(* 
 * Default GetQName
 *)
GetQName[___] := {}
  
(* 
 * GetType for XMLElement with type defined in an attribute as a List
 *)
GetType[
  XMLElement[
    (_String | {_String, _String}), 
    {___, 
     {_String?SchemaInstanceNamespaceQ, "type"}->type:{_String,_String}, 
     ___}, 
    {___}], 
  _List, 
  ___] := type

(* 
 * GetType for XMLElement with type defined in an attribute as a String
 *) 
GetType[
  XMLElement[
    (_String | {_String, _String}), 
    {___, 
     {_String?SchemaInstanceNamespaceQ, "type"}->type_String, 
     ___}, 
    {___}], 
  namespaces_List, 
  ___] := GetType[type, namespaces]

(* 
 * GetType for XMLElement with type not defined in an attribute, so 
 * the default is used.
 *) 
GetType[
  XMLElement[
    (_String | {_String, _String}), 
    {___}, 
    {___}], 
  _List,
  default:{_String, _String}] := default

(* 
 * GetType for XMLElement syntax.  The name is used to determine the type. 
 *)
GetType[
  XMLElement[
    name:{_String, _String}, 
    {___}, 
    {___}],  
  _List,
  ___] := name

(* 
 * GetType for XMLElement syntax.  The child is used to determine the type. 
 *)
GetType[
  XMLElement[
    (_String | {_String, _String}), 
    {___}, 
    {val_String}],  
  _List,
  ___] := {"http://www.w3.org/2001/XMLSchema", "string"}
  
(* 
 * GetType for Rule syntax with type defined in an attribute as a List
 *)
GetType[
  ({(_String | {_String, _String}), 
     {___, 
      {_String?SchemaInstanceNamespaceQ, "type"}->type:{_String,_String}, 
      ___}}->_), 
  _List,
  ___] := type

(* 
 * GetType for Rule syntax with type defined in an attribute as a String
 *)
GetType[
  ({(_String | {_String, _String}), 
     {___, 
      {_String?SchemaInstanceNamespaceQ, "type"}->type_String, 
      ___}}->val_), 
  namespaces_List,
  ___] := GetType[type, namespaces]

(* 
 * GetType for Rule syntax with type not defined in an attribute, so 
 * the default is used.
 *) 
GetType[
  ((_String | {(_String | {_String, _String}), 
    {___}})->_), 
  _List,
  default:{_String, _String}] := default

(* 
 * GetType for Rule syntax.  The child is used to determine the type.
 *)
GetType[
  ((_String | {(_String | {_String, _String}), 
    {___}})->val_), 
  _List,
  ___] := 
  Module[{},
    Switch[
      val,
      True | False, {"http://www.w3.org/2001/XMLSchema", "boolean"},
      _String, {"http://www.w3.org/2001/XMLSchema", "string"},
      _Integer, {"http://www.w3.org/2001/XMLSchema", "int"},
      _Real, {"http://www.w3.org/2001/XMLSchema", "double"},
      _, {}
    ]
  ]

(*
 * GetType where the name is used to determine the type for 
 * complex objects 
 *)
GetType[
  ({name:{_String, _String}, {___}}->___), 
   _List,
   {_String, _String}] := name

(* 
 * GetType for Rule syntax definition with type defined in an 
 * attribute as a List 
 *)
GetType[
  ({(_String | {_String, _String}), 
     {___, 
      {_String?SchemaInstanceNamespaceQ, "type"}-> type:{_String, _String}, 
      ___}}), 
  _List, 
  ___] := type
  
(* 
 * GetType for Rule syntax definition with type defined in an 
 * attribute as a String 
 *)
GetType[
  ({(_String | {_String, _String}), 
     {___, 
      {_String?SchemaInstanceNamespaceQ, "type"}->type_String, 
      ___}}), 
  namespaces_List,
  ___] := GetType[type, namespaces]

(* 
 * GetType for QName String representation
 *)
 
GetType[t_String?WrappedQNameQ, namespaces_List,___] := 
  GetQName[t, namespaces]
 
(* 
 * GetType for prefixed string value 
 *) 
 
GetType[t_String?PrefixedQNameQ, namespaces_List,___] :=
  GetQName[t, namespaces] 

(* 
 * QName for Strings without a prefix or namespace
 *) 
GetType[t_String, namespaces_List, targetNamespace_String] := {targetNamespace, t}
 
(* 
 * QName for Strings without a prefix or namespace
 *) 
GetType[t_String, namespaces_List,___] := {"", t}

(* 
 * Default GetType,  Could perhaps return something more useful or 
 * something that conforms to {_String, _String}
 *)
GetType[___] := {}

(* 
 * GetName for XMLElements
 *)
GetName[
  XMLElement[
    (name_String | name:{_String, _String}), 
    {___}, 
    {___}]] := name

(* 
 * GetName for Rule syntax
 *)
GetName[
  (name_String | {(name_String | name:{_String, _String}), 
    {___}})->_] := name

(* 
 * GetName for Rule syntax definition 
 *)
GetName[
  {(name_String | name:{_String, _String}), 
    {___}}] := name
 
GetName[___] := Null

(* 
 * GetQualifedName for XMLElements for a List
 *)
GetQualifiedName[XMLElement[name:{_String, _String}, {___}, {___}]] := name

(* 
 * GetQualifiedName for XMLElements for a String
 *)
GetQualifiedName[XMLElement[name_String, {___}, {___}]] := {"", name}

(* 
 * GetQualifiedName for Rule syntax for a List
 *)
GetQualifiedName[{name:{_String, _String}, {___}}->_] := name

(* 
 * GetQualifiedName for Rule syntax for a String
 *)
GetQualifiedName[(name_String | {name_String, {___}})->_] := {"", name}

(* 
 * GetQualifiedName for Rule syntax definition for a List
 *)
GetQualifiedName[{name:{_String, _String}, {___}}] := name

(* 
 * GetQualifiedName for Rule syntax definition for a String
 *)
GetQualifiedName[{name_String, {___}}] := {"", name}

GetQualifiedName[___] := {}

(* 
 * GetLocalName for XMLElements
 *)
GetLocalName[
  XMLElement[
    (name_String | {_String, name_String}), 
    {___},
    {___}]] := name

(* 
 * GetLocalName for Rule syntax
 *)
GetLocalName[
  (name_String | {(name_String | {_String, name_String}), 
    {___}})->_] := name

(* 
 * GetLocalName for Rule syntax definition
 *)
GetLocalName[
  (name_String | {(name_String | {_String, name_String}), {___}})] := name

GetLocalName[___] := Null

(* 
 * GetAttribute for XMLElements for attributes with namespace
 *)
GetAttribute[
  namespace_String,
  localpart_String,
  XMLElement[
    (_String | {_String, _String}), 
    attrs:{___}, 
    {___}]] :=
  Module[{cases},
    cases = Select[attrs, MatchQ[#, {namespace, localpart}->_]&];
    If[cases === {}, Null, cases]
  ]

(* 
 * GetAttribute for XMLElements for attributes without a namespace
 *)
GetAttribute[
  localpart_String,
  XMLElement[
    (_String | {_String, _String}), 
    attrs:{___}, 
    {___}]] :=
  Module[{cases},
    cases = Select[attrs, MatchQ[#, (localpart | {_String, localpart})->_]&];
    If[cases === {}, Null, cases]
  ]

(* 
 * GetAttribute for Rule syntax for attributes with namespace
 *)
GetAttribute[
  namespace_String,
  localpart_String,
  {(_String | {_String, _String}), attrs:{___}}->_] :=
  Module[{cases},
    cases = Select[attrs, MatchQ[#, {namespace, localpart}->_]&];
    If[cases === {}, Null, cases]
  ]

(* 
 * GetAttribute for Rule syntax for attributes without a namespace
 *)
GetAttribute[
  localpart_String,
  {(_String | {_String, _String}), attrs:{___}}->_] :=
  Module[{cases},
    cases = Select[attrs, MatchQ[#, (localpart | {_String, localpart})->_]&];
    If[cases === {}, Null, cases]
  ]

(* 
 * GetAttribute for Rule syntax definition for attributes with namespace
 *)
GetAttribute[
  namespace_String,
  localpart_String,
  {(_String | {_String, _String}), attrs:{___}}] :=
  Module[{cases},
    cases = Select[attrs, MatchQ[#, {namespace, localpart}->_]&];
    If[cases === {}, Null, cases]
  ]

(* 
 * GetAttribute for Rule syntax definition for attributes without a namespace
 *)
GetAttribute[
  localpart_String,
  {(_String | {_String, _String}), attrs:{___}}] :=
  Module[{cases},
    cases = Select[attrs, MatchQ[#, (localpart | {_String, localpart})->_]&];
    If[cases === {}, Null, cases]
  ]

GetAttribute[___] := Null

(* 
 * GetAttributes for XMLElements
 *)
GetAttributes[XMLElement[(_String | {_String, _String}), 
                         attrs:{___}, 
                         {___}]] := attrs

(* 
 * GetAttributes for Rule syntax
 *)
GetAttributes[{(_String | {_String, _String}), attrs:{___}}->_] := attrs


(* 
 * GetAttributes for Rule syntax definition
 *)
GetAttributes[{(_String | {_String, _String}), attrs:{___}}] := attrs

GetAttributes[___] := {}

(* 
 * GetValue for XMLElements
 *)
GetValue[
  XMLElement[
    (_String | {_String, _String}), 
    {___}, 
    value:{___}]] := value
                         
(* 
 * GetValue for Rule syntax
 *)
GetValue[{(_String | {_String, _String}), {___}}->value_] := value

GetValue[___] := Null

(* 
 * Used for converting Rule syntax to symbolic XML
 *) 
GetSymbolicXML[
  (name_String | {(name_String | name:{_String, _String}), 
     attrs:{___}})->val_] :=
  Module[{newAttrs = attrs, newVal = val},
    If[Head[Unevaluated[attrs]] =!= List, newAttrs = {}];
    If[!ListQ[val], newVal = {val}];
    newVal = GetSymbolicXML /@ newVal;
    XMLElement[name, newAttrs, newVal]
  ]

GetSymbolicXML[x___] := ToString[x]

GetPattern[{namespace_?BaseTypeNamespaceQ, name_String}]:=
  Switch[name,
    "int", _Integer,
    "integer", _Integer,
    "double", _Real,
    "float", _Real,
    "string", _String,
    "boolean", (True | False),
    "short", _Integer,
    "long", _Integer,
    "decimal", _Real,
    "byte", _Integer,
    "nonPositiveInteger", _Integer,
    "negativeInteger", _Integer,
    "nonNegativeInteger", _Integer,
    "unsignedLong", _Integer,
    "unsignedInt", _Integer,
    "unsignedShort", _Integer,
    "unsignedByte", _Integer,
    "positiveInteger", _Integer,
    "duration", _String,
    "dateTime", _String, 
    "time", _String,
    "date", _String,
    "gYearMonth", _String,
    "gYear", _String,
    "gMonthDay", _String,
    "gDay", _String,
    "gMonth", _String,
    "base64Binary", _String,
    "hexBinary", _String,
    "anyURI", _String,
    "QName", _String,
    "normalizedString", _String,
    "token", _String,
    "language", _String,
    "Name", _String, 
    "NCName", _String,
    "NMTOKEN", _String,
    "NMTOKENS", _String,
    "ID", _String, 
    "IDREF", _String,
    "IDREFS", _String,
    "ENTITY", _String,
    "ENTITIES", _String,
    "base64", _String,
    _, _
  ];

GetPattern[___] := _;

(* Used for mapping namespaces contained in the Namespaces option and attributes.  This is needed
   because the type attribute generally contained a prefixed qualified name.  The Namespaces option
   is used to look up the prefix.  The attributes defining a namespace are found and added the 
   option (which can be set at runtime.)
 *)

namespaceSameQ[{namespace1_String, local1_String}->_, {namespace2_String, local2_String}->_] :=
  (namespace1 === namespace2 && local1 === local2);
  
namespaceSameQ[___] := False;

MapNamespaces[
  XMLElement[
    (_String | {_String, _String}), 
    attrs:{___}, 
    {___}], 
  namespaces_List] :=
  Join[Select[attrs, MatchQ[#, {_?XMLNamespaceQ, _String}->_String]&], namespaces]

MapNamespaces[
  ((name_String | {(name_String | {ns_String, name_String}), 
     attrs:{___}})->_), 
  namespaces_List] :=
  If[Head[Unevaluated[attrs]] === List, 
    Join[Select[attrs, MatchQ[#, {_?XMLNamespaceQ, _String}->_String]&], namespaces],               
    namespaces
  ]

MapNamespaces[
  ((name_String | {(name_String | {ns_String, name_String}), 
     attrs:{___}})), 
  namespaces_List] :=
  If[Head[Unevaluated[attrs]] === List, 
    Join[Select[attrs, MatchQ[#, {_?XMLNamespaceQ, _String}->_String]&], namespaces],               
    namespaces
  ] 
  
MapNamespaces[_, namespaces_List] := namespaces;

MapNamespaces[___] := {}
  
(*
 * Function used to clean up Strings used to make Symbols for InstallService and
 * InstallServiceOperation.
 *)     
NormalizeSymbolName[name_String] :=
  StringReplace[name, {"."->"", "_"->"", "~"->"", "!"->"", 
                       "@"->"", "#"->"", "$"->"", "%"->"", "^"->"", 
                       "&"->"", "*"->"", "("->"", ")"->"", "-"->"", 
                       "+"->"", "="->"", "{"->"", "["->"", "}"->"", 
                       "]"->"", "|"->"", "\\"->"", ":"->"", ";"->"",
                       "\""->"", "\'"->"", "<"->"", ","->"", ">"->"",
                       "?"->"", "/"->"", " "->""}]
     
GetSchema[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "definitions"}, 
    {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/", "types"}, 
       {___}, 
        children:{___}], 
        ___}],
    schemaName_String] :=     
  Module[{results}, 
    results = 
      Cases[
        children, 
        XMLElement[
          {_?SchemaNamespaceQ, "schema"}, 
          {___, {"", "targetNamespace"}->schemaName, ___}, 
          {___}]];
    If[Length[results] > 0, First[results], Null]
  ];

GetSchema[___] := Null;

getTypeContent[
  type:{namespace_String, name_String}, 
  s_
  ]:=
  Module[{nspaces, typeSchema, schema = s, defaultForm},
    nspaces = MapNamespaces[schema, {}];
    If[!SchemaQ[schema], 
      schema = GetSchema[schema, namespace];
      nspaces = MapNamespaces[schema, nspaces];
    ]; 
    defaultForm = DefaultQualifiedFormQ[schema];
    typeSchema = GetTypeDef[name, schema];
    nspaces = MapNamespaces[typeSchema, nspaces];
    {type, typeSchema, nspaces, defaultForm}
  ]

GetTypeContent[type:{namespace_String, _String}] :=
  Module[{result = {type, Null, {}, False}, r = {element, Null, {}}},
    If[MatchQ[r = getTypeContent[type, #], {type, Null, {}, _}] === False,
      result = r; 
    ] & /@ GetSchemaDefinitions[namespace];
    result
  ]
  
GetTypeContent[___] := {Null, Null, Null, Null};
     
getElementContent[
  element:{namespace_String, name_String},
  s_
  ]:=
  Module[{nspaces, elementSchema, ref, schema = s, defaultForm},
        
    nspaces = MapNamespaces[schema, {}];
    If[!SchemaQ[schema], 
      schema = GetSchema[schema, namespace];
      nspaces = MapNamespaces[schema, nspaces];
    ];
    defaultForm = DefaultQualifiedFormQ[schema];
    elementSchema = GetElementDef[name, schema];
    nspaces = MapNamespaces[elementSchema, nspaces];
    If[(ref = GetElementRef[elementSchema]) =!= Null,
      GetElementContent[GetQName[ref, nspaces]], 
      {element, elementSchema, nspaces, defaultForm}
    ]
  ]

GetElementContent[element:{namespace_String, _String}] :=
  Module[{result = {element, Null, {}, False}, r = {element, Null, {}}},
    If[MatchQ[r = getElementContent[element, #], {_, Null, {___}, _}] === False,
      result = r
    ] & /@ GetSchemaDefinitions[namespace];
    result
  ]

GetElementContent[___] := {Null, Null, Null, Null};

GetArrayType[type:XMLElement[
  {_?SchemaNamespaceQ, "complexType"}, 
  {___}, 
  children:{___}]] :=
  Module[{results, elements, maxOccurs}, 
    
    results = 
      Cases[children, 
        XMLElement[
          {_?SchemaNamespaceQ, "attribute"}, 
          {___, 
           {"http://schemas.xmlsoap.org/wsdl/","arrayType"}->arrayType_String,
           ___}, 
          {___}] :> arrayType, Infinity];
    If[Length[results] > 0, Return[First[results]]];
    (*
    elements = GetElements[type];
    If[Length[elements] =!= 1, Return[Null]];

    maxOccurs = GetElementMaxOccurs[First[elements]];
    If[maxOccurs === Null, Return[Null]];
    If[maxOccurs === "unbounded", 
      Return[GetElementType[First[elements]]]
    ];
    If[DigitQ[maxOccurs] && ToExpression[maxOccurs] > 1, 
      Return[GetElementType[First[elements]]]
    ];
    *)
    Null
  ];      

GetArrayType[___] := Null;

IsExpression[
  XMLElement[
    {_?SchemaNamespaceQ, "complexType"},
    {___},
    {___,
      XMLElement[
        {_?SchemaNamespaceQ, ("sequence" | "all")},
        {___},
        {element:XMLElement[
          {_?SchemaNamespaceQ, "element"}, 
          {___, {"", "ref"}->{"http://www.wolfram.com/XML/", "Expression"}, ___}, 
          {___}]
        }],
    ___}],
  nspaces_List] := True;
  
IsExpression[
  XMLElement[
    {_?SchemaNamespaceQ, "complexType"},
    {___},
    {___,
      XMLElement[
        {_?SchemaNamespaceQ, ("sequence" | "all")},
        {___},
        {element:XMLElement[
          {_?SchemaNamespaceQ, "element"}, 
          {___, {"", "ref"}->val_String, ___}, 
          {___}]
        }],
    ___}],
  nspaces_List] := 
  Module[{qname},
    qname = GetQName[val, nspaces];
    MatchQ[qname, {"http://www.wolfram.com/XML/", "Expression"}]
  ];

IsExpression[___] := False;
     
End[]