 (* :Author: Chris Williamson *)

Begin["`Package`"];

MapWSDL::usage = 
  "MapWSDL is a utility function used to map a WSDL file into Mathematica functions.";

GetWSDLDefinitions::usage =
  "GetWSDLDefinitions is a utility function that provides access to WSDL definitions.";

End[];

Begin["`WSDL`Private`"]

InstallService::"context" = 
	"Illegal Context: `1`, `2`"	
	
InstallService::"symbol" = 
	"Illegal Symbol (begins with a digit): `1`"	
	
InstallService::"style" = "Style not supported: `1`"

InstallService::"dfntn" = 
	"The Definition element is not found for this WSDL"

InstallService::"elmnt" = 
	"Element not defined in part: `1`"
	
InstallService::"prefix" = 
	"Problem finding prefix for `1`"
	
InstallService::"prttpop" = 
  "The port type operation is Null.";
  
Needs[ "JLink`" ];

(* Retrieve definitions and store conveniently in memory *)
getDef[
  XMLObject["Document"][
    {___}, 
    def:XMLElement[
      {"http://schemas.xmlsoap.org/wsdl/", "definitions"},
      {___}, 
      {___}], 
    {___}]] := def;

getDef[___] := Null;

(* Retrieve targetNamespace and store conveniently in memory *)
getTargetNS[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "definitions"},
    {___, {"", "targetNamespace"}->targetNamespace_String, ___}, 
    {___}]] := 
  targetNamespace;
                     
getTargetNS[___] := Null;

getServices[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "definitions"}, 
    {___}, 
    children:{___}]] :=
  Cases[children, XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "service"}, {___}, {___}]];
  
getServices[___] := {};


getImports[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "definitions"}, 
    {___}, 
    children:{___}]] :=
  Cases[children, XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "import"}, {___}, {___}]];

getImports[___] := {};

getImportNamespace[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "import"}, 
    {___,{"", "namespace"}->namespace_String,___}, 
    {___}]] := namespace;
  
getImportNamespace[___] := Null;

getImportLocation[
  XMLElement[
  {"http://schemas.xmlsoap.org/wsdl/", "import"}, 
  {___,{"", "location"}->location_String,___}, 
  {___}]] := location;
  
getImportLocation[___] := Null;

getSchemas[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "definitions"}, 
    {___}, 
    {___, 
     types:XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/", "types"}, 
       {___}, 
       children:{___}],
     ___}]] :=
  Cases[children, XMLElement[{_?SchemaNamespaceQ, "schema"}, {___}, {___}]];

getSchemas[___] := Null;

addSchemaDefinitions[
  XMLElement[
    {_?SchemaNamespaceQ, "include"},
    {___, {"", "schemaLocation"}->location_String, ___},
    {___}],
    parentURL_String] :=

  Module[{targetNamespace, includes, symbolicXML, schema, url,
          host, relative, loc},
    If[$PrintWSDLDebug === True,    
      Print["Processing schemaLocation: ", location]
    ];
    
    JavaBlock[
      url = JavaNew["java.net.URL", JavaNew["java.net.URL", parentURL], location];
      loc = url@toString[];
    ];
    
    symbolicXML = XML`Parser`XMLGet[loc];      
    symbolicXML = XML`ToVerboseXML[symbolicXML];    
    schema = getSchema[symbolicXML];
    targetNamespace = GetTargetNamespace[schema];
    If[targetNamespace === Null, 
     Message[InstallServiceOperation::"import", location];
     Return[$Failed]
    ];

    If[$PrintWSDLDebug === True, 
      Print["Adding Schema Definitions: ", targetNamespace]
    ];
    GetSchemaDefinitions[targetNamespace] = 
      AppendTo[GetSchemaDefinitions[targetNamespace], schema];
    includes = GetIncludes[schema];
    addSchemaDefinitions[#, location] & /@ includes;    
  ];

addSchemaDefinitions[
  schema:XMLElement[
    {_?SchemaNamespaceQ, "schema"},
    {___, {"", "targetNamespace"}->targetNamespace_String,___},
    {___}],
  def:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "definitions"},
    {___},
    {___}],
  parentURL_String] :=
   
  Module[{includes},
    GetSchemaDefinitions[targetNamespace] = 
      AppendTo[GetSchemaDefinitions[targetNamespace], def];
    includes = GetIncludes[schema];
    addSchemaDefinitions[#, parentURL] & /@ includes;          
  ]

addSchemaDefinitions[
  def:XMLElement[
    defName:{"http://schemas.xmlsoap.org/wsdl/", "definitions"},
    defAttrs:{___},
    {___, 
     XMLElement[
       typesName:{"http://schemas.xmlsoap.org/wsdl/", "types"}, 
       typesAttrs:{___}, 
       {___}],
     ___}],
  parentURL_String] :=  
  Module [{schemas},
    schemas = getSchemas[def];
    addSchemaDefinitions[
      #, 
      XMLElement[
        defName, 
        defAttrs, 
        {XMLElement[
           typesName, 
           typesAttrs,
           {#}]}],
      parentURL] & /@ schemas;
   ]

getSchema[
  XMLObject["Document"][
    {___}, 
    schema:XMLElement[{_?SchemaNamespaceQ, "schema"}, {___}, {___}], 
    {___}]] := schema;
  
getSchema[___] := Null;

setImport[
  import:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "import"}, 
    {___}, 
    {___}]] :=
       
  Module[{namespace, location, symbolicXML},
    namespace = getImportNamespace[import];
    If[namespace === Null, 
      Message[InstallServiceOperation::"import", Null];
      Return[$Failed]
    ];
    
    location = getImportLocation[import];
    If[location === Null, 
      Message[InstallServiceOperation::"import", namespace];
      Return[$Failed]
    ];
      
    symbolicXML = XML`Parser`XMLGet[location];
    If[symbolicXML === $Failed, 
      Message[InstallServiceOperation::"import", namespace];
      Return[$Failed]
    ];
      
    symbolicXML = XML`ToVerboseXML[symbolicXML];
    If[symbolicXML === $Failed, 
      Message[InstallServiceOperation::"import", namespace];
      Return[$Failed]
    ];
    
    If[getDef[symbolicXML] =!= Null, 
      GetWSDLDefinitions[namespace] = getDef[symbolicXML],
      If[symbolicXML === $Failed, 
        Message[InstallServiceOperation::"import", namespace];
        Return[$Failed]
      ];
    ];
  ];

MapWSDL[
  ServicesImpl["Java"],
  wsdlURL_String, 
  options___?OptionQ] := MapWSDL[ServicesImpl["Java"], wsdlURL, Null, options]

MapWSDL[
  ServicesImpl["Java"],
  wsdlURL_String, 
  context_,
  options___?OptionQ] :=

  Module[{positions, test, symbolicXML, definition, targetNamespace, 
          schemas, imports, services, namespaces, results},
    
      (* Check to make sure the user-specified context is legal *)
      If[context =!= Null, 
 
        (* Context must end with a '`' *)
        If[!StringMatchQ[context, "*`"], 
          Message[
            InstallService::"context", 
            context, 
            "contexts must end with a '`'"];
          Return[$Failed];
        ];
  
        (* Context must not contain illegal characters *)
        If[!MatchQ[
             StringPosition[
               context, 
               {".", "_", "~", "!", "@", "#", "$", "%", "^", 
                "&", "*", "(", ")", "-", "+", "=", "{", "[", 
                "}", "]", "|", "\\", ":", ";", "\"", "\'", 
                "<", ",", ">", "?", "/", " "}], 
             {}], 
          Message[
            InstallService::"context", 
            context, 
            "contexts must be alpha-numeric."];
          Return[$Failed];
        ];
 
        (* Contexts must not begin with a digit *)
        positions = 
          Drop[Prepend[(First[#] + 1) & 
            /@ StringPosition[context, "`"], 1], -1];
        test = (If[DigitQ[StringTake[context, {#,#}]], $Failed] & /@ positions);
        If[Length[Cases[test, $Failed]] > 0, 
          Message[
            InstallService::"context", 
            context, 
            "contexts must not begin with a digit."];
          Return[$Failed];
        ];
      ];
      getContext[] = context;
      If[$PrintWSDLDebug === True, Print["Context: ", getContext[]]];
      
      getWsdlUrl[] = wsdlURL;
      If[$PrintWSDLDebug === True, Print["WSDL URL: ", getWsdlUrl[]]];

      getOptions[] = {options};
      If[$PrintWSDLDebug === True, Print["Options: ", getOptions[]]];
        
      symbolicXML = XML`Parser`XMLGet[wsdlURL];
      If[symbolicXML === $Failed, Return[$Failed]];
      
      symbolicXML = XML`ToVerboseXML[symbolicXML];
      If[symbolicXML === $Failed, Return[$Failed]];
          
      definition = getDef[symbolicXML];
      If[definition === Null, 
        Message[InstallService::"dfntn"]; Return[$Failed]
      ];
  
      targetNamespace = getTargetNS[definition];
      If[$PrintWSDLDebug === True, Print["Target Namespace: ", targetNamespace]];

      GetWSDLDefinitions[targetNamespace] = definition;
      
      addSchemaDefinitions[definition, wsdlURL];
      
      imports = getImports[definition];
      setImport /@ imports;
     
      services = getServices[definition];
      
      namespaces = MapNamespaces[definition, {}];
      results = 
        Select[
          Flatten[
            mapService[#, MapNamespaces[#, namespaces]] & /@ services], 
          # =!= Null &];
      
      Unset[getContext[]];
      Unset[getWsdlUrl[]];
      Unset[getOptions[]];
      Unset[GetWSDLDefinitions];
      
      results
    ];

getPorts[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "service"}, 
    {___}, 
    children:{___}]] :=
  Cases[children, XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "port"}, {___}, {___}]];
  
getPorts[___] := {};

getDocumentation[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "service"}, 
    {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/", "documentation"}, 
       {___}, 
       {doc_String}],
     ___}]] := doc;

getDocumentation[___] := Null;

mapService[
  service:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "service"}, 
    {___, {"", "name"}->name_String,___}, 
    {___}],
  namespaces_List] := 

  Module[{ports, results},
  
    ports = getPorts[service];

    getServiceName[] = name;
    If[getContext[] === Null,  
      (* Check to make sure it is a legal name.  Drop illegal characters. *)
      getServiceName[] = NormalizeSymbolName[name];
                                      
      (* Check to make sure portName does not begin with a number *)
      If[DigitQ[StringTake[name, 1]], 
        Message[InstallService::"symbol", name]; 
        Return[$Failed];
      ];      
      If[$PrintWSDLDebug === True, Print["Service Name: ", getServiceName[]]];
    ];

    getServiceDocumentation[] := getDocumentation[service];

    results = mapPort[#, MapNamespaces[#, namespaces]] & /@ ports;
    
    Unset[getServiceName[]];
    Unset[getServiceDocumentation[]];
    
    results 
  ];

mapService[___] := $Failed;

getBindingName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "port"}, 
    {___, {"", "binding"}->binding_String, ___}, 
    {___}]] := binding;

getBindingName[___] := Null;

getEndPoint[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "port"}, 
    {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/soap/", "address"}, 
       {___, {"", "location"}->location_String, ___}, 
       {___}], 
     ___}]] := location

getEndPoint[___] := Null;

getBinding[
  bindingName_String, 
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/","definitions"},
    {___}, 
    children:{___}]] := 

  Module[{results},
    results = 
      Cases[
        children, 
        XMLElement[
          {"http://schemas.xmlsoap.org/wsdl/", "binding"}, 
          {___, {"", "name"} -> bindingName, ___}, 
          {___}]];
    If[Length[results] > 0, First[results], Null]
  ];
        
getBinding[___] := Null;

getSOAPBinding[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "binding"}, 
    {___}, 
    {___, 
     soapBinding:XMLElement[
      {"http://schemas.xmlsoap.org/wsdl/soap/", "binding"}, 
      {___}, 
      {___}], 
     ___}]] := soapBinding;
  
getSOAPBinding[___] := Null;

getGlobalTransport[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "binding"}, 
    {___, {"", "transport"}->transport_String, ___}, 
    {___}]] := transport;

getGlobalTransport[___] := "http://schemas.xmlsoap.org/soap/http";

getGlobalStyle[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "binding"}, 
    {___, {"", "style"}->style_String, ___}, 
    {___}]] := style;

getGlobalStyle[___] := "document";

getBindingOperations[binding:XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "binding"}, 
                       {___}, children:{___}]] := 
  Cases[children, XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "operation"}, {___}, {___}]];
  
getBindingOperations[___] := Null;

getPortTypeName[port:XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "binding"}, 
                 {___, {"", "type"}->portType_String, ___}, {___}]] := portType;

getPortTypeName[___] := Null;

getPortType[portTypeName_String, def:XMLElement[{"http://schemas.xmlsoap.org/wsdl/","definitions"},
            {___},children:{___}]] := 
  Module[{results},
    results = Cases[children, XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "portType"}, 
                     {___, {"", "name"} -> portTypeName, ___}, {___}]];
    If[Length[results] > 0, First[results], Null]
  ];
        
getPortType[___] := Null;

mapPortTypeOperations[portTypeOp:XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
                       {___,{"", "name"}->name_String,___}, {___}]] :=
  getPortTypeOperation[name] = portTypeOp;

mapPort[port:XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "port"}, 
          {___, {"", "name"}->name_String, ___}, {___}],
        namespaces_List] :=
  Module[{bindingName, binding, soapBinding, portTypeName, nspaces, ptNspaces,
          bindingOperations, portTypeOperations, qname, portType, results}, 

    getPortName[] = name;
    If[getContext[] === Null,  
      getPortName[] = NormalizeSymbolName[name];
                                      
      If[DigitQ[StringTake[getPortName[], 1]], 
        Message[InstallService::"symbol", getPortName[]]; 
        Return[$Failed];
      ];      
      If[$PrintWSDLDebug === True, Print["Port Name: ", getPortName[]]];    
    ];  

    getSOAPEndPoint[] = getEndPoint[port];
    (* This quietly exits if there is no SOAP binding. *)
    If[getSOAPEndPoint[] === Null, Return[]];
    If[$PrintWSDLDebug === True, Print["SOAP End Point: ", getSOAPEndPoint[]]];
    
    bindingName = getBindingName[port];
    qname = GetQName[bindingName, namespaces];
    If[!MatchQ[qname, {_String, _String}], 
      Message[InstallService::"prefix", bindingName];
      Return[$Failed];
    ];
    binding = getBinding[Last[qname], GetWSDLDefinitions[First[qname]]];    
    nspaces = MapNamespaces[GetWSDLDefinitions[First[qname]], {}];
    nspaces = MapNamespaces[binding, nspaces];

    soapBinding = getSOAPBinding[binding];
    getTransport[] = getGlobalTransport[soapBinding];
    If[$PrintWSDLDebug === True, Print["Transport: ", getTransport[]]];

    getStyle[] = getGlobalStyle[soapBinding];    
    If[getStyle[] === Null || 
       (!StringMatchQ[getStyle[], "rpc"] && 
        !StringMatchQ[getStyle[], "document"]), 
      Message[InstallService::"style", getStyle[]];
      Return[$Failed];
    ];
    If[$PrintWSDLDebug === True, Print["Style: ", getStyle[]]];

    bindingOperations =  getBindingOperations[binding];
    
    portTypeName = getPortTypeName[binding];
    qname = GetQName[portTypeName, namespaces];
    If[!MatchQ[qname, {_String, _String}], 
      Message[InstallService::"prefix", portTypeName];
      Return[$Failed];
    ];
    portType = getPortType[Last[qname], GetWSDLDefinitions[First[qname]]];
    ptNspaces = MapNamespaces[GetWSDLDefinitions[First[qname]], {}];
    ptNspaces = MapNamespaces[portType, ptNspaces];
    
    results = 
      mapOperation[#, MapNamespaces[#, nspaces], portType, ptNspaces] & 
        /@ bindingOperations;
    
    Unset[getPortName[]];
    Unset[getSOAPEndPoint[]];
    Unset[getTransport[]];
    Unset[getStyle[]];
    
    results
  ];

mapPort[___] := $Failed;

getInputName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "input"}, 
    {___,{"", "name"}->inputName_String,___} , 
    {___}]] := inputName;

getInputName[___] := Null;
                              
getOutputName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "output"}, 
    {___,{"", "name"}->outputName_String,___} , 
    {___}]] := ouputName;

getOutputName[___] := Null;

opEqualQ[
  op1:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___, {"", "name"} -> opName1_String, ___}, 
    {___}],
  op2:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___, {"", "name"} -> opName2_String, ___}, 
    {___}]] := 
  (If[(opName1 === opName2) && 
      (((getOutputName[getOutput[op1]] === getOutputName[getOutput[op2]]) && 
        (getInputName[getInput[op1]] === getInputName[getInput[op2]])) ||
       (getOutputName[getOutput[op2]] === Null && getInputName[getInput[op2]] == Null)), True, False]);
  
opEqualQ[___] := False;

getPortTypeOperation[
  bindingOp:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___}, 
    {___}],
  portType:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "portType"}, 
    {___}, 
    children:{___}]] := 
  Module[{results},
    results = Select[children, opEqualQ[#, bindingOp] &];
    If[Length[results] > 0, First[results], Null]
  ];
  
getPortTypeOperation[___] := Null;

setOption[options_List, option_?OptionQ] := 
  Module[{opts},
    If[(First[option] /. options) === First[option],
      opts = Append[options, option],
      opts = options /. {Rule[First[option],_]->option}
    ]
  ]

mapOperation[bindingOp:XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
               {___, {"", "name"}->name_String, ___}, {___}],
             namespaces_List,
             portType:XMLElement[{"http://schemas.xmlsoap.org/wsdl/", "portType"}, 
                       {___}, children:{___}],
             ptNspaces_List] :=
  Module[{opName, portTypeOp, params, returnType, sym, options, opts, doc = Null, nspaces}, 
      
    If[$PrintWSDLDebug === True, Print["Operation Name: ", name]];
    
    mapBindingOperation[bindingOp, namespaces];
      
    getOperation[] = Null;
    If[StringMatchQ[getStyle[], "rpc"], 
      getOperation[] = {getNamespaceURI[], name};
      If[$PrintWSDLDebug === True, Print["Operation: ", getOperation[]]];
    ];

    portTypeOp = getPortTypeOperation[bindingOp, portType];
    If[portTypeOp === Null, Message[InstallService::"prttpop"];Return[$Failed]];
    nspaces = MapNamespaces[portTypeOp, ptNspaces];

    params = mapPortTypeOperation[portTypeOp, nspaces];
    If[params === $Failed, Return[$Failed]];
    If[$PrintWSDLDebug === True, Print["Params: ", params]];

    returnType = mapReturnType[portTypeOp, nspaces];
    If[returnType === $Failed || returnType === Null, returnType = Automatic];
    If[$PrintWSDLDebug === True, Print["Return Type: ", returnType]];
   
    options = getOptions[];
    
    opts = {
      OperationName -> getOperation[],
      OperationStyle -> getStyle[],
      SOAPActionURI -> getSOAPActionURI[],
      ReturnType -> returnType,
      TransportStyleURI -> getTransport[],
      EncodingStyleURI -> getEncodingStyleURI[],
      EncodingStyle->getEncodingStyle[],
      DeencodeParameters->If[!StringMatchQ[getEncodingStyle[], "encoded"], True, False] 

    };
    
    (opts = setOption[opts, #] ) &  /@ options;
    
    opName = NormalizeSymbolName[name];
    If[DigitQ[StringTake[name, 1]], 
      Message[InstallService::"symbol", opName]; 
      Return[$Failed];
    ];

    If[getContext[] =!= Null, 
      sym = getContext[] <> opName,
      sym = getServiceName[] <> "`" <> getPortName[] <> "`" <> opName;
    ];
    
    sym = ToExpression[sym];
    InstallServiceOperation[
      sym, 
      getSOAPEndPoint[], 
      params, 
      getHeaderParams[], 
      opts];

    AddMethodToPalette[
      ServicesImpl["Java"], 
      sym, 
      getDocumentation[], 
      params, 
      getHeaderParams[], 
      returnType];
    
    With[{op = sym, wsdl = getWsdlUrl[]},
      GetOperationPalette[sym] := 
        GetOperationPalette[sym] = 
          GeneratePalette[ServicesImpl["Java"], op, wsdl]
    ];
    
    doc = "Documentation was not provided.";             
    If[getDocumentation[] =!= Null && StringQ[getDocumentation[]], 
      doc = getDocumentation[]
    ];
    If[$PrintWSDLDebug === True, Print["Documentation: ", doc]];
        
    MessageName[Evaluate[sym], "usage"] = 
      doc <> 
        "\n\nPlease refer to \!\(\*ButtonBox[\"OperationPalette[" <> 
          ToString[sym] <> "]\",
          Active->True, 
          ButtonEvaluator->Automatic,
          ButtonFunction->OperationPalette[" <> ToString[sym] <> "],
          ButtonStyle->\"Hyperlink\"]\) for help.";
        
    Unset[getSOAPActionURI[]];
    Unset[getEncodingStyleURI[]];
    Unset[getNamespaceURI[]];
    Unset[getEncodingStyle[]];
    Unset[getBindingParts[]];
    Unset[getOperation[]];
    Unset[getHeaderParams[]];
    
    sym
  ];

mapOperation[___] := $Failed;

getActionURI[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/soap/", "operation"}, 
       {___, {"", "soapAction"}->soapAction_String, ___}, 
       {___}], 
     ___}]] := soapAction;

getActionURI[___] := Null;

getInput[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___}, 
    {___, 
     input:XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/", "input"}, 
       {___}, 
       {___}], 
     ___}]] := input;
  
getInput[___] := Null;

getSOAPHeaders[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "input"}, 
    {___}, 
    children:{___}]] := 
  Cases[
    children, 
    XMLElement[
      {"http://schemas.xmlsoap.org/wsdl/soap/", "header"}, 
      {___}, 
      {___}]];

getSOAPHeaders[___] := {}

getEncodingStyle[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "input"}, 
    {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/soap/", "body"}, 
       {___, {"", "encodingStyle"}->encodingStyle_String,___}, 
       {___}], 
     ___}]] := encodingStyle;

getEncodingStyle[___] := "";

getNamespace[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "input"}, 
    {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/soap/", "body"}, 
       {___, {"", "namespace"}->namespace_String,___}, 
       {___}], 
     ___}]] := namespace;

getNamespace[___] := "";

getLocalEncodingStyle[
  input:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "input"}, 
    {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/soap/", "body"}, 
       {___, {"", "use"}->use_String,___}, 
       {___}], 
     ___}]] := use;

getLocalEncodingStyle[___] := "literal";


getBindParts[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "input"}, {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/soap/", "body"}, 
       {___, {"", "parts"}->parts_String,___}, 
       {___}], 
     ___}]] := parts;

getBindParts[___] := Null;

getMessageName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "header"}, 
    {___, {"", "message"}->messageName_String, ___}, 
    {___}]] := messageName;

getPartName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "header"}, 
    {___, {"", "part"}->partName_String, ___}, 
    {___}]] := partName;

getPartName[___] := Null;

getPart[
  partName_String, 
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/","message"},
    {___},
    children:{___}]] := 
  Module[{results},
    results = 
      Cases[
        children, 
        XMLElement[
          {"http://schemas.xmlsoap.org/wsdl/", "part"}, 
          {___, {"", "name"} -> partName, ___}, 
          {___}]];
    If[Length[results] > 0, First[results], Null]
  ];

getPart[___] := Null;

getLocalEncodingStyle[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "header"}, 
    {___, {"", "use"}->use_String, ___}, 
    {___}]] := use;

getEncodingStyle[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "header"}, 
    {___, {"", "encodingStyle"}->encodingStyle_String, ___}, 
    {___}]] := encodingStyle;

getNamespace[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "header"}, 
    {___, {"", "namespace"}->namespace_String, ___}, 
    {___}]] := namespace;


mapHeader[
  header:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/soap/", "header"}, 
    {___}, 
    {___}],
  namespaces_List] :=
  
  Module[{messageName, message, partName, part, ns, 
          headerEncodingStyle, nspaces, qname},

    messageName = getMessageName[header];
    qname = GetQName[messageName, namespaces];
    If[!MatchQ[qname, {_String, _String}], 
      Message[InstallService::"prefix", messageName];
      Return[$Failed];
    ];
    message = getMessage[Last[qname], GetWSDLDefinitions[First[qname]]];
    nspaces = MapNamespaces[GetWSDLDefinitions[First[qname]], {}];
    nspaces = MapNamespaces[message, nspaces];
    
    partName = getPartName[header];
    part = getPart[partName, message];
    nspaces = MapNamespaces[part, nspaces];
    
    (*getNamespaceURI[] = getNamespace[header];*)
    
    headerParam = mapPart[part, First[qname], nspaces, True]
  ]

mapBindingOperation[
  bindingOp:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___}, 
    {___}],
  namespaces_List] :=
  
  Module[{bindingInput, headers, encodingStyle, nspaces},
  
    getSOAPActionURI[] = getActionURI[bindingOp];
    If[$PrintWSDLDebug === True, 
      Print["SOAP Action URI: ", getSOAPActionURI[]]
    ];    
    
    bindingInput = getInput[bindingOp];
    nspaces = MapNamespaces[bindingInput, namespaces];

    getHeaderEncodingStyleURI[] = Automatic;
    headers = getSOAPHeaders[bindingInput];
    getHeaderParams[] = (mapHeader[#, MapNamespaces[#, nspaces]] & /@ headers);
    If[$PrintWSDLDebug === True, Print["Header Params: ", getHeaderParams[]]];
      
    getEncodingStyleURI[] = ImportString[getEncodingStyle[bindingInput], "Words"];
    If[getEncodingStyleURI[] === {}, getEncodingStyleURI[] = Automatic];
    If[$PrintWSDLDebug === True, Print["Encoding Style URI: ", getEncodingStyleURI[]]];
    
    getNamespaceURI[] = getNamespace[bindingInput];
    If[$PrintWSDLDebug === True, Print["Namespace URI: ", getNamespaceURI[]]];
    
    getEncodingStyle[] = getLocalEncodingStyle[bindingInput];
    If[$PrintWSDLDebug === True, Print["Encoding Style: ", getEncodingStyle[]]];

    getBindingParts[] = getBindParts[bindingInput];
    If[getBindingParts[] =!= Null, 
      getBindingParts[] = ExportString[getBindingParts[], "Words"]
    ];
    If[$PrintWSDLDebug === True, Print["Binding Parts: ", getBindingParts[]]];    
  ]

mapBindingOperation[___] := $Failed;

getMessageName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", ("input"|"output")}, 
    {___, {"", "message"}->messageName_String, ___}, 
    {___}]] := messageName;
                 
getMessageName[___] := Null;

getMessage[
  messageName_String, 
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/","definitions"},
    {___},
    children:{___}]] := 
  Module[{results},
    results = 
      Cases[
        children, 
        XMLElement[
          {"http://schemas.xmlsoap.org/wsdl/", "message"}, 
          {___, {"", "name"} -> messageName, ___}, 
          {___}]];
    If[Length[results] > 0, First[results], Null]
  ];
        
getMessage[___] := Null;

getParts[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "message"}, 
    {___}, 
    children:{___}]] :=
  Cases[
    children, 
    XMLElement[
      {"http://schemas.xmlsoap.org/wsdl/", "part"}, 
      {___}, 
      {___}]];
  
getParts[___] := {}
              
samePartQ[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "part"}, 
    {___, {"", "name"} -> partName_String, ___}, 
    {___}], 
  name_String] := 
  MatchQ[partName, name];

samePartQ[___] := False;

getDocumentation[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___}, 
    {___, 
     XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/", "documentation"}, 
       {___}, 
       {doc_String}],
     ___}]] := doc;

mapPortTypeOperation[
  portTypeOp:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___}, 
    {___}],
  namespaces_List] :=
  
  Module[{doc = Null, portTypeInput, messageName, message, parts, 
          params, nspaces, qname},

    doc = getDocumentation[portTypeOp];
    If[doc =!= Null, 
      getDocumentation[] = doc, 
      getDocumentation[] = getServiceDocumentation[]
    ];
    
    portTypeInput = getInput[portTypeOp];
    nspaces = MapNamespaces[portTypeInput, namespaces];
    
    messageName = getMessageName[portTypeInput];
    qname = GetQName[messageName, nspaces];
    If[!MatchQ[qname, {_String, _String}], 
      Message[InstallService::"prefix", messageName];
      Return[$Failed];
    ];
    message = getMessage[Last[qname], GetWSDLDefinitions[First[qname]]];    
    If[message === Null, Return[{}]];
    nspaces = MapNamespaces[GetWSDLDefinitions[First[qname]], {}];
    nspaces = MapNamespaces[message, nspaces];
    
    parts = getParts[message];
    If[getBindingParts[] =!= Null, 
      parts = Intersection[parts, getBindingParts[], SameTest->samePartQ];
    ];

    If[getStyle[] === "document",
      If[parts === {}, 
        params = {}, 
        params = mapPart[First[parts], getNamespaceURI[], nspaces, False]
      ],
      params = mapPart[#, getNamespaceURI[], nspaces, False] & /@ parts;
    ];
    
    params

  ];

mapPortTypeOperation[___] := $Failed;

getTypeName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "part"}, 
    {___, {"", "type"}->type_String, ___}, 
    {___}]] := type;

getTypeName[___] := Null;

getElementName[
  XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "part"}, 
    {___, {"", "element"}->element_String, ___}, 
    {___}]] := element;

getElementName[___] := Null;

setArrayType[type:{_String, _String}, namespaces_List] := 
  Module[{t, schemaNamespaces, typeSchema, arrayType, defaultForm},
    {t, typeSchema, schemaNamespaces, defaultForm} = GetTypeContent[type];    

    If[(arrayType = GetArrayType[typeSchema]) =!= Null,
      {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->
        GetType[arrayType, schemaNamespaces],
      Null
    ]    
  ];

setArrayType[
  type:XMLElement[
    {_?SchemaNamespaceQ, ("complexType" | "simpleType")}, 
    {___}, 
    {___}],
  namespaces_List] :=

  Module[{arrayType},
    If[(arrayType = GetArrayType[type]) =!= Null,
      {"http://schemas.xmlsoap.org/soap/encoding/","arrayType"}->
        GetType[arrayType, namespaces],
      Null
    ]        
  ];

setArrayType[___] := Null;

mapAttributes[
  type:XMLElement[
    {_?SchemaNamespaceQ, ("simpleType" | "complexType")}, 
    {___,{"", "name"}->name_String,___}, 
    children:{___}], 
  targetNamespace_String,
  namespaces_List, 
  header:(True | False)] :=
  
  Module[{attrs = {}, arrayTypeAttr},
  
    (* Add type if needed. *)
    AppendTo[
      attrs, 
      {"http://www.w3.org/2001/XMLSchema-instance","type"}->
        {targetNamespace, name}];
      
    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[type, namespaces]) =!= Null, 
      AppendTo[attrs, arrayTypeAttr]
    ];
    
    If[TrueQ[header],
      AppendTo[
        attrs, 
        {"http://schemas.xmlsoap.org/soap/envelope/","actor"}->""];
      AppendTo[
        attrs, 
        {"http://schemas.xmlsoap.org/soap/envelope/","mustUnderstand"}->"0"];
    ];   
    attrs
  ]

mapAttributes[
  type:{_String, _String}, 
  targetNamespace_String,
  namespaces_List,
  header:(True | False)] :=
  Module[{attrs = {}, arrayTypeAttr},
  
    (* Add type if needed. *)
    AppendTo[attrs, {"http://www.w3.org/2001/XMLSchema-instance","type"}->type];
      
    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[type, namespaces]) =!= Null, 
      AppendTo[attrs, arrayTypeAttr]
    ];
    
    If[TrueQ[header],
      AppendTo[
        attrs, 
        {"http://schemas.xmlsoap.org/soap/envelope/","actor"}->""];
      AppendTo[
        attrs, 
        {"http://schemas.xmlsoap.org/soap/envelope/","mustUnderstand"}->"0"];
    ];
    attrs
  ]


mapAttributes[
  typeName_String, 
  targetNamespace_String, 
  namespaces_List, 
  header:(True | False)] :=
  
  Module[{type, attrs = {}, arrayType},

    type = GetType[typeName, namespaces, targetNamespace];
    If[!MatchQ[type, {_String, _String}], 
      Message[InstallService::"prefix", typeName];
      Return[{}];
    ];    
    
    (* Add type if needed. *)
    AppendTo[attrs, {"http://www.w3.org/2001/XMLSchema-instance","type"}->type];
      
    (* set array type *)    
    If[(arrayTypeAttr = setArrayType[type, namespaces]) =!= Null, 
      AppendTo[attrs, arrayTypeAttr]
    ];
    
    If[TrueQ[header],
      AppendTo[
        attrs, 
        {"http://schemas.xmlsoap.org/soap/envelope/","actor"}->""];
      AppendTo[
        attrs, 
        {"http://schemas.xmlsoap.org/soap/envelope/","mustUnderstand"}->"0"];
    ];
    attrs
  ]

mapAttributes[___] := {}

resolveReferences[
  XMLElement[
    {_?SchemaNamespaceQ, "element"}, 
    {___, {"", "ref"}->ref_String, ___}, 
    {___}], 
  typeNS_String,
  nspaces_List,
  defaultForm_] :=
  GetElementContent[GetQName[ref, nspaces]];
  

resolveReferences[
  x:XMLElement[
    {_?SchemaNamespaceQ, "element"}, 
    {___}, 
    {___}], 
  typeNS_String, 
  nspaces_List,
  defaultForm_] := {{typeNS, GetElementName[x]}, x, nspaces, defaultForm};

mapParam[
  {{namespace_String, name_String},
   element:XMLElement[
     {_?SchemaNamespaceQ, "element"}, 
     {___}, 
     {___}],  
   nspaces_List,
   defaultForm_},
  header:(True | False)] := 
  Module[{ns = ""},
    (* Place the correct namespace for local elements.    
       If it is qualified it should take the targetNamespace.  
       Otherwise it should be "" *)
    If[QualifiedFormQ[element, defaultForm], ns = namespace];
    {{ns, name}, mapAttributes[GetElementType[element], namespace, nspaces, header]}    
  ];

mapPart[
  part:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "part"}, 
    {___,{"", "name"}->name_String,___}, 
    {___}], 
  namespace_String,
  namespaces_List,
  header:(True | False)] :=
  
  Module[{typeName, type = Null, elementName, elementNS = namespace, defaultForm,  
          element = Null, schema, params, qname, nspaces, typeSchema, elementSchema, ref,
          targetNS = namespace},
       
    nspaces = MapNamespaces[#, namespaces];
    
    typeName = getTypeName[part];
    If[typeName =!= Null, 
      type = GetType[typeName, namespaces];
      If[MatchQ[type, {_String, _String}], 
        {type, typeSchema, nspaces, defaultForm} = GetTypeContent[type];
        targetNS = First[type];
        If[typeSchema === Null, typeSchema = type],
        Message[InstallService::"prefix", typeName];
      ];
    ];
    
    elementName = getElementName[part];
    If[elementName =!= Null, 
      element = GetQName[elementName, namespaces];
      If[MatchQ[element, {_String, _String}], 
        {element, elementSchema, nspaces, defaultForm} = GetElementContent[element];
        If[elementSchema =!= Null,
          typeSchema = GetElementType[elementSchema];
          targetNS = First[element];
          nspaces = MapNamespaces[typeSchema, nspaces];
          If[StringQ[typeSchema], 
            type = GetType[typeSchema, namespaces];
            If[MatchQ[type, {_String, _String}], 
              {type, typeSchema, nspaces, defaultForm} = GetTypeContent[type];
              targetNS = First[type];
              If[typeSchema === Null, typeSchema = type],
              Message[InstallService::"prefix", typeName];
            ];
          ],
          Message[InstallService::"elmnt", elementName];
        ],
        Message[InstallService::"prefix", elementName];
      ];
    ];
    If[getStyle[] === "rpc", 
      Return[{{"", name}, mapAttributes[typeSchema, targetNS, nspaces, header]}];
    ];
    If[header === False,
      If[getStyle[] === "document",
        getOperation[] = element;
        If[$PrintWSDLDebug === True, Print["Operation: ", getOperation[]]];  
        params = GetElements[typeSchema];
        params = resolveReferences[#, targetNS, nspaces, defaultForm] & /@ params;
        Return[mapParam[#, header]& /@ params];
      ],
      Return[{{namespace, name}, mapAttributes[typeSchema, targetNS, nspaces, header]}];
    ];
    $Failed
  ];

mapPart[___] := $Failed;
    
getOutput[
  operation:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___}, 
    {___, 
     input:XMLElement[
       {"http://schemas.xmlsoap.org/wsdl/", "output"}, 
       {___}, 
       {___}], 
     ___}]] := input;
  
getOutput[___] := Null;

mapReturnType[
  portTypeOp:XMLElement[
    {"http://schemas.xmlsoap.org/wsdl/", "operation"}, 
    {___}, 
    {___}],
  namespaces_List] :=

  Module[{portTypeOutput, messageName, message, parts, part, qname, defaultForm,
          type, elementName, element, nspaces, typeSchema, elementSchema,
          schemaNspaces, return},
  
    portTypeOutput = getOutput[portTypeOp];
    If[portTypeOutput === Null, Return[Null]];

    messageName = getMessageName[portTypeOutput];
    qname = GetQName[messageName, namespaces];
    If[!MatchQ[qname, {_String, _String}], 
      Message[InstallService::"prefix", messageName];
      Return[$Failed];
    ];
    message = getMessage[Last[qname], GetWSDLDefinitions[First[qname]]];    
    If[message === Null, Return[{}]];
    nspaces = MapNamespaces[GetWSDLDefinitions[First[qname]], {}];
    nspaces = MapNamespaces[message, nspaces];
    
    parts = getParts[message];
        
    If[parts === Null,
      Return[Null],
      If[Length[parts] > 0, 
        part = First[parts],
        Return[Null];
      ];
    ];
    
    nspaces = MapNamespaces[part, nspaces];
    
    typeName = getTypeName[part];
    If[typeName =!= Null, 
      type = GetType[typeName, nspaces];
      If[MatchQ[type, {_String, _String}], 
        typeNS = First[type];
        {type, typeSchema, schemaNspaces, defaultForm} = GetTypeContent[type];
        If[typeSchema === Null, typeSchema = type],
        Message[InstallService::"prefix", typeName];
      ];
    ];
    
    elementName = getElementName[part];
    If[elementName =!= Null, 
      element = GetQName[elementName, nspaces];
      If[MatchQ[element, {_String, _String}], 
        {element, elementSchema, schemaNspaces, defaultForm} = GetElementContent[element];
        If[elementSchema =!= Null,
          typeSchema = GetElementType[elementSchema];
          typeNS = First[element];
          schemaNspaces = MapNamespaces[typeSchema, schemaNspaces];
          If[StringQ[typeSchema], 
            type = GetType[typeSchema, namespaces];
            If[MatchQ[type, {_String, _String}], 
              typeNS = First[type];
              {type, typeSchema, schemaNspaces, defaultForm} = GetTypeContent[type];
              If[typeSchema === Null, typeSchema = type],
              Message[InstallService::"prefix", typeName];
            ];
          ],
          Message[InstallService::"elmnt", elementName];
        ],
        Message[InstallService::"prefix", elementName];
      ];
    ];

    If[getStyle[] === "rpc", 
      If[type =!= Null && type =!= {}, 
        Return[type],
        Return[Null]
      ];
    ];
    If[getStyle[] === "document",
      elements = GetElements[typeSchema];
      If[elements =!= Null, 
        If[Length[elements] > 0, 
          element = First[elements];
          {element, elementSchema, schemaNspaces, defaultForm} = resolveReferences[element, typeNS, nspaces, defaultForm];
          typeNS = First[element];
          elementType = GetElementType[elementSchema];
          If[elementType =!= Null,
            If[StringQ[elementType],
              Return[GetType[elementType, schemaNspaces, typeNS]],
              If[(elementType = GetTypeName[elementType]) =!= Null, 
                Return[{typeNS, elementType}],
                Return[Null];
              ];
            ],
            Return[Null];
          ];
        ];
      ];
    ];
    Null
  ];

mapReturnType[___] := Null;
      
End[];