(* :Title: CallJava.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 3.1 *)

(* :Mathematica Version: 4.0 *)
		     
(* :Copyright: J/Link source code (c) 1999-2005, Wolfram Research, Inc. All rights reserved.

   Use is governed by the terms of the J/Link license agreement, which can be found at
   www.wolfram.com/solutions/mathlink/jlink.
*)

(* :Discussion:
   Functions related to loading classes and calling from Mathematica into Java.
	
   This file is a component of the J/Link Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   J/Link uses a special system wherein one package context (JLink`) has its implementation
   split among a number of .m files. Each component file has its own private context, and also
   potentially introduces public symbols (in the JLink` context) and so-called "package" symbols,
   where the term "package" comes from Java terminology, referring to symbols that are visible
   everywhere within the implementation of J/Link, but not to clients. The JLink.m file itself
   is produced by an automated tool from the component files and contains only declarations.
   
   Do not modify the special comment markers that delimit Public- and Package-level exports.
*)


(*<!--Public From CallJava.m

JavaObject::usage =
"JavaObject is used to denote an expression that refers to an object residing in Java."

JavaObjectQ::usage =
"JavaObjectQ[expr] gives True if expr is a reference to a Java object or Null, and gives False otherwise."

JavaClass::usage =
"JavaClass[\"classname\", n] represents a Java class with the specified name. The second argument is an integer index that is not relevant to users. JavaClass expressions cannot be typed in by the user; they are created by LoadJavaClass."

LoadClass::usage =
"LoadClass is deprecated. The new name is LoadJavaClass."

LoadJavaClass::usage =
"LoadJavaClass[\"classname\"] loads the specified class into Java and sets up definitions so that it can be used from Mathematica. You must specify the fully-qualified class name, for example \"java.awt.Frame\". It is safe to call LoadJavaClass multiple times on the same class; it simply returns right away without doing anything."

JavaNew::usage =
"JavaNew[javaclass, args] constructs a Java object of the specified JavaClass, passing the args to its constructor. You can also specify the class name instead of a JavaClass: JavaNew[\"classname\", args]. With this second form, the class will be loaded first if necessary."

MethodFunction::usage =
"MethodFunction[javaclass, method] returns a function that can be used to invoke the given method on objects of the given class. You can also specify the class by its fully-qualified name or by an object of the class, as in MethodFunction[\"classname\", method] or MethodFunction[javaobject, method]. mf = MethodFunction[cls, meth] followed by mf[obj, args] is identical to the more usual form obj@meth[args]. MethodFunction is purely an optimization--if you are executing the same method call hundreds or thousands of times you can use a MethodFunction to avoid repeating the extra internal steps necessary with the obj@meth[args] syntax. MethodFunction is mostly obsolete. Speed improvements in J/Link have made its benefits minimal."

FieldFunction::usage =
"FieldFunction[javaclass, field] returns a function that can be used to get or set the value of the given field in objects of the given class. You can also specify the class by its fully-qualified name or by an object of the class, as in FieldFunction[\"classname\", field] or FieldFunction[javaobject, field]. ff = FieldFunction[cls, meth] followed by ff[obj] is identical to the more usual form obj@field. FieldFunction is purely an optimization--if you are accessing the same field hundreds or thousands of times you can use a FieldFunction to avoid repeating the extra internal steps necessary with the obj@meth[args] syntax. FieldFunction is mostly obsolete. Speed improvements in J/Link have made its benefits minimal."

SetField::usage =
"SetField[obj@field, val] sets a value of an object field. It is an alternative syntax to obj@field = val. For static methods, use SetField[staticfield, val] (compared to staticfield = val). Supporting the obj@field = val syntax for fields is problematic because it requires rules to be created for the built-in function Set. SetField avoids this at the cost of breaking the simple correspondence between Java and Mathematica syntax."

StaticsVisible::usage =
"StaticsVisible is an option to LoadJavaClass (in J/Link) and LoadNETType (in .NET/Link) that specifies whether the class-specific context in which static method and field definitions are created should be placed on $ContextPath. If you load a class named \"com.foo.Bar\" containing a static method named baz, with the option StaticsVisible->True you could call the method simply with baz[args]. With StaticsVisible->False, you would have to write Bar`baz[args]. The default setting is False, to eliminate the possibility of name conflicts leading to shadowing problems."

AllowShortContext::usage =
"AllowShortContext is an option to LoadJavaClass (in J/Link) and LoadNETType (in .NET/Link) that specifies whether to create a class-specific context in \"short\" form. If you load a class named \"com.foo.Bar\" containing a static method named baz, with the option AllowShortContext->True you could call the method with Bar`baz[args]. With AllowShortContext->False, you would have to write com`foo`Bar`baz[args]. AllowShortContext->False allows you to avoid conflicts with other contexts in the system. The default is True."

UseTypeChecking::usage =
"UseTypeChecking is an option to LoadJavaClass that specifies whether to include or omit the type checking performed in Mathematica on arguments to Java calls. The default is UseTypeChecking->True. UseTypeChecking is largely superseded by the flag $RelaxedTypeChecking. If you use UseTypeChecking->False, you must make sure you pass methods exactly the argument types they expect. Also note that J/Link will not be able to correctly differentiate between multiple definitions of the same method that take the same number of arguments. Most users will have no use for this option."

Val::usage =
"Val is deprecated. The new name is JavaObjectToExpression."

JavaObjectToExpression::usage =
"JavaObjectToExpression[javaobject] converts the specified Java object reference into its value as a \"native\" Mathematica expression. Normally, all Java objects that have a meaningful \"by value\" representation in Mathematica are returned by value to Mathematica automatically. Such objects include strings, arrays (these become lists), and instances of so-called wrapper classes like java.lang.Integer. However, you can get a reference form of one of these types if you explicitly call JavaNew or use the ReturnAsJavaObject function. In such cases, you can use JavaObjectToExpression to retrieve the value. JavaObjectToExpression has no effect on objects that have no meaningful \"value\" representation in Mathematica."

ByRef::usage =
"ByRef is deprecated. The new name is ReturnAsJavaObject."

ReturnAsJavaObject::usage =
"ReturnAsJavaObject[expr] causes a Java method call or field access during the evaluation of expr to return its result as an object reference (that is, a JavaObject epxression), not a value. Most Java objects are returned as references normally, but those that have a meaningful Mathematica representation are returned \"by value\". Such objects include strings, arrays, and the so-called wrapper classes like java.lang.Integer. ReturnAsJavaObject overrides the normal behavior and forces any object returned to Mathematica to be sent only as a reference. It is typically used to avoid needlessly sending large arrays of numbers back and forth between Java and Mathematica. You can use ReturnAsJavaObject to cause only a reference to be sent, then use the JavaObjectToExpression function at the end if the final value is needed."

-->*)

(*<!--Package From CallJava.m

// Used in jlinkExternalCall to direct output to the appropriate link (JavaLink[] or JavaUILink[]).
getActiveJavaLink

// These 5 are called directly from Java.
jlinkDefineExternal
loadClassAndCreateInstanceDefs
createInstanceDefs
loadClassFromJava
issueNoDefaultCtorMessage

// Exported so that defs created as classes/objects are loaded can be cleared when Java is restarted.
clearJavaDefs
callAllUnloadClassMethods

// Various methods for converting among class/name/index.
classIndexFromInstance
classNameFromIndex
classFromIndex
parentClassIndexFromClassIndex
classNameFromClass
classIndexFromClass

-->*)


(* Current context will be JLink`. *)

Begin["`CallJava`Private`"]


(****************************************  Java constants  *********************************************)

(* Must remain in sync with Java code. *)
TYPEBOOLEAN		= -1
TYPEBYTE		= -2
TYPECHAR		= -3
TYPESHORT		= -4
TYPEINT			= -5
TYPELONG		= -6
TYPEFLOAT		= -7
TYPEDOUBLE		= -8
TYPESTRING		= -9
TYPEBIGINTEGER	= -10
TYPEBIGDECIMAL	= -11
TYPEEXPR		= -12
TYPECOMPLEX		= -13
(* Every type that has a byval representation in Mathematica must be > TYPEOBJECT
   (required only for createArgPairs). This does not apply to TYPE_DOUBLEORINT and
   TYPE_FLOATORINT, which are used in a very limited way.
*)
TYPEOBJECT		= -14
TYPEFLOATORINT	= -15
TYPEDOUBLEORINT	= -16
TYPEARRAY1		= -17
TYPEARRAY2		= TYPEARRAY1 + TYPEARRAY1
TYPEARRAY3		= TYPEARRAY2 + TYPEARRAY1
TYPEBAD			= -10000


(******************************************  callJava  ***********************************************)

(* callJava is a wrapper function that exists for two purposes:

   1) to turn off preemptive functions in a 5.1+ kernel. The JavaLink[] is a "Service link",
      meaning that it is used for calls that originate in M at unexpected times. We always need
      to turn off preemption when using a service link because a preemptive call could try
      to reentrantly use a service link.
   2) to interpose the ReturnAsJavaObject logic before calling jCallJava.

   All definitions in this file are made in terms of callJava.
   If $byRef is True, we alter the args to force a "by ref" return before sending them along to jCallJava.
*)

callJava[{a__, 1}, b___] /; $byRef := jCallJava[{a, 0}, b]
callJava[c_, d___]                 := jCallJava[c, d]


(******************************************  JavaNew  ************************************************)

JavaNew::argx0 = "There is no constructor for class `1` that takes zero arguments."
JavaNew::argx1 = "Incorrect number or type of arguments to constructor for class `1`. The argument was `2`."
JavaNew::argx = "Incorrect number or type of arguments to constructor for class `1`. The arguments, shown here in a list, were `2`."
JavaNew::fail = "Error calling constructor for class `1`."
JavaNew::intf = "Cannot create object of type `1`, because it is an interface, not a class."
JavaNew::invcls =
"The JavaClass expression `1` is not valid in the current Java session. The Java runtime may have been quit and restarted since this expression was defined."
JavaNew::defctor =
"Cannot use JavaNew to create an array of objects of class `1`, because this class does not have a no-argument constructor."

JavaNew[cls_String, args___] :=
	Module[{index = classIndexFromName[cls], jc},
		If[Head[index] =!= Integer,
			jc = LoadJavaClass[cls];
			If[Head[jc] === JavaClass,
				JavaNew[classIndexFromClass[jc], args],
			(* else *)
				(* LoadJavaClass will have already issued a message if it failed. *)
				$Failed
			],
		(* else *)
			JavaNew[index, args]
		]
	]

JavaNew[cls_JavaClass, args___] :=
	If[isLoadedClass[cls],
		JavaNew[classIndexFromClass[cls], args],
	(* else *)
		Message[JavaNew::invcls, cls];
		$Failed
	]

(* Users will not call this def directly. *)
JavaNew[index_Integer, args___] :=
	Module[{res},
		If[interfaceQFromIndex[index],
			Message[JavaNew::intf, classNameFromIndex[index]];
			Return[$Failed]
		];
		res = javaNew[index, args];
		If[JavaObjectQ[res],
			res,
		(* else *)
			If[Head[res] === javaNew,
				Switch[Length[{args}],
					0,
						Message[JavaNew::argx0, classNameFromIndex[index]],
					1,
						Message[JavaNew::argx1, classNameFromIndex[index], args],
					_,
						Message[JavaNew::argx, classNameFromIndex[index], {args}]
				],
			(* else *)
				Message[JavaNew::fail, classNameFromIndex[index]]
			];
			$Failed
		]
	]
	
(* All the defs for JavaNew funnel down to the form with an integer (classIndex) as the first arg. This def calls
    javaNew, which only has defs created for it during LoadJavaClass. All these defs do nothing but call javaConstructor:
   
	   javaNew[index_Integer, argPats...] := javaConstructor[index, {1,1,1}, argtype, arg1, argtpe, arg2,...]
	   
    Thus, javaNew exists only for pattern-matching. It's a bottleneck that ensures that only legit arg sequences
    make it through to javaConstructor, which calls Java directly.
   
    NOTE: Now that I have changed javaConstructor so that it does nothing but call callJava, I can get rid of
    javaConstructor completely, making defs for javaNew call callJava...
*)

javaConstructor[classIndex_Integer, indices_List, argTypesAndVals___] :=
	callJava[{classIndex, 1, Null, indices, 0}, Length[{argTypesAndVals}]/2, argTypesAndVals]


(* This func is called from Java to alert the user that they have tried to
   do JavaNew of an array class, the components of which are objects without a no-arg ctor.
*)
issueNoDefaultCtorMessage[cls_String] := Message[JavaNew::defctor, cls]


(********************************************  SetField  ****************************************************)

(* SetField is the preferred method for setting fields. The alternative is the natural syntax obj@field = val. Supporting
   that requires a hack to Set (below). This slows down Set and is a bit fragile. Probably should use SetField myself,
   except possibly in simple user-visible demos. I might want to eliminate the Set hack in the future and require SetField.
   
   Note that ALL field setting, even statics, is routed through setField. The top layer of providing a means to use Set notation
   is just syntactic sugar. I could easily eliminate all reliance on Set syntax for fields. In the case of static fields, there
   is no motivation to do this since Set method has no drawbacks (for statics, we use upvalues on symbols for Set behavior, not
   downvalues on Set).
*)

SetAttributes[SetField, HoldFirst]
SetField[sym_[field_Symbol], val_] := With[{obj = sym}, setField[obj[field], val]]
SetField[staticField_Symbol, val_] := setField[staticField, val]

SetAttributes[setField, HoldFirst]
(* These are the only rules for setField, fall-throughs that issue a message. Other uses are via UpValues on objects or static symbols. *)
setField[obj_[_], _] := (Message[Java::obj, obj]; $Failed)
setField[sym_Symbol, _] := (Message[Java::flds, HoldForm[sym]]; $Failed)

(* Cost of this Set hack is that it makes Set run a little slower for args that don't match the pattern, about 5 times
   slower for args that do, and f in f[symbol] = val gets evaluated twice. I could eliminate this without breaking anything
   in this package. Would then have to always use SetField for object-based field sets (i.e., not statics).
*)
prot = Unprotect[Set]
Set[sym_Symbol[arg_Symbol], val_] := 
	With[{obj = sym},
		setField[obj[arg], val] /; Head[obj] === Symbol && Context[obj] === "JLink`Objects`"
	]
Protect[Evaluate[prot]]

(*****
Here is a version of the Set hack that doesn't eval f in f[symbol] twice. The cost is that it runs 10 times slower than
unaltered Set for args that match the pattern:

Set[sym_Symbol[arg_Symbol], val_] := 
	Block[{inSetPatch = True},
		With[{obj = sym},
			If[Head[obj] === Symbol && Context[obj] === "JLink`Objects`",
				setField[obj[arg], val],
			(* else *)
				Set[obj[arg], val]
			]
		]
	] /; !TrueQ[inSetPatch]
*****)


(********************************  MethodFunction and FieldFunction  **************************************)

Java::obj = "Attempting to use invalid Java object: `1`."
Java::flds = "Attempting to use invalid Java static field: `1`."

Java::argx0 = "Method named `1` defined in class `2` does not take zero arguments."
Java::argx1 = "Method named `1` defined in class `2` was called with an incorrect number or type of arguments. The argument was `3`."
Java::argx = "Method named `1` defined in class `2` was called with an incorrect number or type of arguments. The arguments, shown here in a list, were `3`."
Java::argxs0 = "The static method `1` does not take zero arguments."
Java::argxs1 = "The static method `1` was called with an incorrect number or type of arguments. The argument was `2`."
Java::argxs = "The static method `1` was called with an incorrect number or type of arguments. The arguments, shown here in a list, were `2`."
Java::nometh = "No method named `1` defined in class `2`."
Java::nometh$ = "No method named `1` defined in class `2`. The method name you specified might have had a conflict with a local variable name in a Module."
Java::fldx = "Attempting to set field named `1` defined in class `2` to an incorrect type of value: `3`."
Java::fldxs = "Attempting to set static field `1` to an incorrect type of value: `2`."
Java::nofld = "No field named `1` defined in class `2`."
Java::nofld$ = "No field named `1` defined in class `2`. The field name you specified might have had a conflict with a local variable name in a Module."
(* This one doesn't really belong here, but it has nowhere else to go. It is issued by MathListener classes
   when user tries to call setHandler[] with an event type that is not supported.
*)
Java::nohndlr = "No event handler method named `1` in `2`."
JavaObject::bad = "The method or field named `1` was called on an invalid Java object."

(* These functions, javaMethod and javaField, are only used during calls that involve an object, static or not.
    That is, the calls that don't go through these functions are static methods and fields that are called by
    name, without an object. These functions allow the "scoping" of names by an object's class, necessary for the
    obj@method[args] syntax to work. You can think of javaMethod and javaField as the bridge between the user-level
    syntax for calling methods/fields, and the internal defs set up for each method/field during LoadJavaClass.
   
    Every 'obj@' call gets through to javaMethod and javaField--there is no pattern-matching that has to be satisfied.
    Here is where I want to catch the errors:
   		- No method/field with that name at all
   		- Bad arg count or type (obj@meth[] was called and it returned unevaluated)
   		
   	We separate the process of constructing a function that knows how to access a given method or field from the process
   	of calling that function on an instance and some args. That way, the optimizations MethodFunction and FieldFunction
   	can perform the first step ahead of time. The instanceFunc function creates the function that will act on an instance.
   	javaField and javaMethod just apply the function that instanceFunction returns to an instance and args.
*)

Attributes[javaMethod] = {HoldAllComplete}
Attributes[javaField] = {HoldAllComplete}
Attributes[instanceFunc] = {HoldAllComplete}

(* TODO: At one point, I wanted to search to see if methName was a defined method for this class, and if not,
    then let the method call evaluate and see if it was a legal method afterwards. This would counteract the
    potentially confusing effects of the HoldAll attribute. Now I'm not sure if this is necessary--could force
    users to Evaluate...
*)
javaMethod[instance_, methName_Symbol, args___] :=
	Block[{instFunc, clsIndex},
		clsIndex = classIndexFromInstance[instance];
		instFunc = lookupInstanceFunc[clsIndex, HoldComplete[methName], True];
		If[Head[instFunc] === Function,
			instFunc,
		(* else *)
			cacheInstanceFunc[clsIndex, HoldComplete[methName], True]
		]
	][instance, args]

(* For 'gets' *)
javaField[instance_Symbol, fieldName_Symbol] :=
	Block[{instFunc, clsIndex},
		clsIndex = classIndexFromInstance[instance];
		instFunc = lookupInstanceFunc[clsIndex, HoldComplete[fieldName], False];
		If[Head[instFunc] === Function,
			instFunc,
		(* else *)
			cacheInstanceFunc[clsIndex, HoldComplete[fieldName], False]
		]
	][instance]

(* For 'sets' *)
javaField[instance_Symbol, fieldName_Symbol, val_] :=
	Block[{instFunc, clsIndex},
		clsIndex = classIndexFromInstance[instance];
		instFunc = lookupInstanceFunc[clsIndex, HoldComplete[fieldName], False];
		If[Head[instFunc] === Function,
			instFunc,
		(* else *)
			cacheInstanceFunc[clsIndex, HoldComplete[fieldName], False]
		]
	][instance, val]

(* MethodFunction and FieldFunction are optimizations available to the user. They "precompile" the time-consuming
   steps of converting the user-level syntax to the internal Class`JPrivate`sym[obj, args] style. This can cut the
   Mathematica-side time in half on a given call, but will only be useful if a method or field is accessed many times.
   The trade-off is that you lose the nice Java-equivalent syntax.
   		mf = MethodFunction[cls, methname];
   		mf[obj, arg]
*)
MethodFunction::null = "Object is null."
FieldFunction::null = "Object is null."

SetAttributes[MethodFunction, {HoldRest}]  (* HoldRest because it is benign, and desirable, for the first arg to evaluate. *)
SetAttributes[FieldFunction, {HoldRest}]

MethodFunction[Null, _]									:= (Message[MethodFunction::null]; $Failed&)
MethodFunction[instance_?JavaObjectQ, methName_Symbol]	:= MethodFunction[classIndexFromInstance[instance], methName]
MethodFunction[cls_JavaClass, methName_Symbol]			:= MethodFunction[classIndexFromClass[cls], methName]
MethodFunction[clsName_String, methName_Symbol]			:= MethodFunction[LoadJavaClass[clsName], methName]
MethodFunction[clsIndex_Integer, methName_Symbol]		:= cacheInstanceFunc[clsIndex, HoldComplete[methName], True]

FieldFunction[Null, _]									:= (Message[FieldFunction::null]; $Failed&)
FieldFunction[instance_?JavaObjectQ, fieldName_Symbol]	:= FieldFunction[classIndexFromInstance[instance], fieldName]
FieldFunction[cls_JavaClass, fieldName_Symbol]			:= FieldFunction[classIndexFromClass[cls], fieldName]
FieldFunction[clsName_String, fieldName_Symbol]			:= FieldFunction[LoadJavaClass[clsName], fieldName]
FieldFunction[clsIndex_Integer, fieldName_Symbol]		:= cacheInstanceFunc[clsIndex, HoldComplete[fieldName], False]

(* Calling a method on an object reference and referring to a field are almost identical in implementation. instanceFunc
   implements this shared functionality. It returns a _function_ that takes an instance and args and calls the field or method.
*)
instanceFunc[clsIndex_, name_Symbol, isMethod_] :=
	Module[{nameStr, sym, resultFunc, wasOn1, wasOn2},
		{wasOn1, wasOn2} = (Head[#] =!= $Off &) /@ {General::spell, General::spell1};
		Off[General::spell, General::spell1];
		(* Find name as symbol in its private context. *)
		nameStr = SymbolName[Unevaluated[name]];
		(* nameStr is now the name in short form, without any context header. *)
		sym = findName[nameStr, clsIndex, isMethod];
		If[sym === $Failed,
			(* findName will already have issued an appropriate message. *)
			resultFunc = $Failed&,
		(* else *)		
			(* Call a def for the symbol in its class-specific `JPrivate context. These defs were created during LoadJavaClass.
			   Note that we don't have to send in the class index, as that is known by the definition made in the class's context.
			*)
			If[isMethod,
				(* The 'instance' below is just a do-nothing wrapper function that distinguishes the first arg
				   from an integer in the argument sequence of a static call of the same name.
				*)
				With[{sym = sym},
					resultFunc = checkMethodResult[sym[instance[#1], ##2], clsIndex]&
				],
			(* else *)
				(* is field *)
				With[{sym = sym},
					resultFunc = checkFieldResult[sym[fieldTag, #1, ##2], clsIndex]&
				]
			]
		];
		If[wasOn1, On[General::spell]];
		If[wasOn2, On[General::spell1]];
		resultFunc
	]


(* Caching. *)

lookupInstanceFunc[clsIndex_, heldName_, isMethod_] := $instanceFuncCache[clsIndex, heldName, isMethod]

cacheInstanceFunc[clsIndex_, heldName:HoldComplete[name_], isMethod_] :=
	Block[{f = instanceFunc[clsIndex, name, isMethod]},
		If[f =!= ($Failed&), $instanceFuncCache[clsIndex, heldName, isMethod] = f];
		f
	]


(* These are funcs used in the functions created by MethodFunction and FieldFunction. They wrap the call and perform error-checking. *)

SetAttributes[checkMethodResult, HoldFirst]
SetAttributes[checkFieldResult, HoldFirst]

checkMethodResult[methodCall_, clsIndex_] := 
	Block[{sym = Head[Unevaluated[methodCall]], result, nameStr},
		result = methodCall; (* This is where the actual call occurs. *)
		If[Head[result] === sym,
			(* Didn't match any pattern for that method--bad arg count or type. Note that the message
			   is issued for the original clsIndex, which is the class of the instance, not actualClsIndex,
			   which is the class the method is declared in.
			*)
			nameStr = StringDrop[ToString[sym], Last[{0} ~Join~ Flatten[StringPosition[ToString[sym], "`"]]]];
			Switch[Length[result] - 1,
				0,
					Message[Java::argx0, nameStr, classNameFromIndex[clsIndex]],
				1,
					Message[Java::argx1, nameStr, classNameFromIndex[clsIndex], Last[result]],
				_,
					Message[Java::argx, nameStr, classNameFromIndex[clsIndex], Rest[List @@ result]]
			];
			$Failed,
		(* else *)
			result
		]
	]

checkFieldResult[fieldCall_, clsIndex_] := 
	Block[{sym = Head[Unevaluated[fieldCall]], result, nameStr},
		result = fieldCall; (* This is where the actual call occurs. *)
		If[Head[result] === sym,
			(* Didn't match any pattern for that field--bad arg type. (Was a 'set' call) *)
			nameStr = StringDrop[ToString[sym], Last[{0} ~Join~ Flatten[StringPosition[ToString[sym], "`"]]]];
			Message[Java::fldx, nameStr, classNameFromIndex[clsIndex], Last[result]];
			$Failed,
		(* else *)
			result
		]
	]


(* Finds method or field name as symbol in its private context, recursively checking superclasses.
   name is the short (no context) name. Criterion for finding the symbol is that it have DownValues defined for it.
   Block used for speed only.
*)
findName[name_String, clsIndex_, isMethod_] :=
	Block[{ctxt, sym, pctxt, parentClsIndex, looksLikeModuleVar, curClsIndex = clsIndex},
		pctxt = "JPrivate`" <> name;
		While[True,
			ctxt = contextNameFromClassIndex[curClsIndex];
			If[Head[ctxt] =!= String,
				Message[JavaObject::bad, name];
				Return[$Failed]
			];
			sym = ToExpression[ctxt <> pctxt];
			If[DownValues[Evaluate[sym]] =!= {},
				Return[sym]
			];
			parentClsIndex = parentClassIndexFromClassIndex[curClsIndex];
			If[parentClsIndex === Null,
				(* We've reached the Object class, so no method of that name is even defined in Mathematica.
				   Note that the message is issued for the original class (this is the class of the instance
				   on which the method is being invoked).
				*)
				looksLikeModuleVar = MatchQ[Characters[name], {__, "$", ___?DigitQ}];
				With[{tag = If[isMethod, "nometh", "nofld"]  <> If[looksLikeModuleVar, "$", ""]},
					Message[MessageName[Java, tag], name, classNameFromIndex[clsIndex]]
				];
				Return[$Failed]
			];
			curClsIndex = parentClsIndex
		]
	]


(**********************************************  LoadJavaClass  ***************************************************)

LoadClass = LoadJavaClass  (* LoadClass is deprecated. *)

LoadJavaClass::fail = "Java failed to load class `1`."
LoadJavaClass::ambig = "Ambiguous function in class `1`. `2` has multiple definitions: `3`."
LoadJavaClass::ambigctor = "Ambiguous constructor in class `1`. Multiple definitions: `2`."

Options[LoadJavaClass] = {StaticsVisible->False, AllowShortContext->True, UseTypeChecking->True}

(* Here is the structure of what Java side sends when a class is loaded:
 	{className_String, classIndex_Integer, ctors_List, methods_List, fields_List}
  where:
    ctors is:   {index_Integer, declaration_String, paramTypes___Integer}
    methods is: {index_Integer, isStatic_True|False, declaration_String, retType_String, name_String, paramTypes___Integer}
    fields is:  {index_Integer, isInherited_True|False, isStatic_True|False, type_String, name_String, type_Integer}
*)

(* New in v2.0, LoadJavaClass lets you supply a second argument that is an object reference whose ClassLoader will be used to load
   the named class. This feature is used by calls to putReference() in Java, when they need to call LoadJavaClass to load the
   classes of the objects they are sending to Mathematica. You can use the second argument from Mathematica also, however.
   The only reason for doing this would be if you needed to ensure that a class was loaded by some special ClassLoader.
   The actual circumstances where you would want to do this, and where the two-argument form of LoadJavaClass would be the best
   way, are too obscure to even go into here.
*)

LoadJavaClass[c:{__String}, objSupplyingClassLoader_Symbol:Null, isBeingLoadedAsComplexClass:(True | False):False, opts___?OptionQ] :=
	LoadJavaClass[#, objSupplyingClassLoader, isBeingLoadedAsComplexClass, opts]& /@ c
	
LoadJavaClass[c1_String, c2__String, objSupplyingClassLoader_Symbol:Null, isBeingLoadedAsComplexClass:(True | False):False, opts___?OptionQ] :=
	LoadJavaClass[{c1, c2}, objSupplyingClassLoader, isBeingLoadedAsComplexClass, opts]

LoadJavaClass[className_String, objSupplyingClassLoader_Symbol:Null, isBeingLoadedAsComplexClass:(True | False):False, opts___?OptionQ] := 
	Module[{lc, classIndex, legalClassName, returnedClassName, parentClass, isInterface, alreadyLoaded,
			staticsVisible, allowShortContext, useTypeChecking, ctorRecs, methRecs, fieldRecs,
			classContext, shortClassContext, usingShortContext, wasOn1, wasOn2, wasOn3},
			
		If[Head[JavaLink[]] =!= LinkObject,
			Message[Java::init];
			Return[$Failed]
		];
		
		alreadyLoaded = classFromName[className];
		If[Head[alreadyLoaded] === JavaClass,
			Return[alreadyLoaded]
		];
	
		If[!TrueQ[$isNestedLoad],
			jAddToClassPath[autoClassPath[], True];
			(* For compatibility with J/Link 1.1. *)
			jAddToClassPath[$ExtraClassPath, True]
		];
		
		(* $isNestedLoad's sole purpose is to avoid repeatedly calling AddToClassPath in the above lines as parent
		   classes are recursively loaded. AddToClassPath is comparatively expensive, and autoClassPath and
		   $ExtraClassPath won't change during the loading of a single class (and all its parent classes).
		*)
		Block[{$isNestedLoad = True},
			lc = jLoadClass[className, objSupplyingClassLoader, isBeingLoadedAsComplexClass]
		];

		(* This pattern is {"classname", index, parentCls_JavaClass, isInterface, {ctors}, {methods}, {fields}} *)
		(* Note that the classname we store is the one returned from loadclass, not the name originally
		    specified by user. The one returned from Java is the fully qualified name.
		*)
		If[!MatchQ[lc, {_String, _Integer, Null | _JavaClass, True | False, _List, _List, _List}],
			Message[LoadJavaClass::fail, className];
			Return[$Failed]
		];

		returnedClassName = lc[[1]];
		classIndex = lc[[2]];
		parentClass = lc[[3]];
		isInterface = lc[[4]];
		ctorRecs = lc[[5]];
		methRecs = lc[[6]];
		fieldRecs = lc[[7]];

		{staticsVisible, allowShortContext, useTypeChecking} =
			{StaticsVisible, AllowShortContext, UseTypeChecking} /. Flatten[{opts}] /. Options[LoadJavaClass];

		usingShortContext = TrueQ[allowShortContext];
		
		storeClass[classIndex,
				   createJavaClass[returnedClassName, classIndex],
				   If[# === Null, Null, classIndexFromClass[#]]& @ parentClass,
				   isInterface,
				   usingShortContext
		];

		{wasOn1, wasOn2, wasOn3} = (Head[#] =!= $Off &) /@ {General::shdw, General::spell, General::spell1};
		Off[General::shdw];
		Off[General::spell];
		Off[General::spell1];
		
		classContext = contextNameFromClassIndex[classIndex];
		(* shortClassContext is for statics. If classContext is java`lang`Foo`, shortClassContext is Foo`.
		    Want to make statics available by using just cass name as context, not full package path.
		*)
		If[usingShortContext,
			shortClassContext = shortClassContextFromClassContext[classContext],
		(* else *)
			shortClassContext = classContext
		];

		createConstructorStubs[classIndex, ctorRecs, TrueQ[useTypeChecking]];
		createMethodStubs[classIndex, classContext, shortClassContext, methRecs, TrueQ[useTypeChecking]];
		createFieldStubs[classIndex, classContext, shortClassContext, fieldRecs, TrueQ[useTypeChecking]];

		(* Call the class's onLoadClass method, if it has one. *)
		Begin[shortClassContext];
		callOnLoadClassMethod[classIndex];
		End[];
		
		(* This occurs after caling the onLoadClass method for a reason, although it's not necessarily a good one.
		    Don't want behavior of code executed by onLoadClass to be dependent on StaticsVisible setting. Force that code
		    to always use full context names for statics it wants to refer to. 
		*)
		If[staticsVisible,
			BeginPackage[classContext];
			EndPackage[]
		];

		If[wasOn1, On[General::shdw]];
		If[wasOn2, On[General::spell]];
		If[wasOn3, On[General::spell1]];

		classFromIndex[classIndex]
	]


(***************************************  Constructor Handling  *********************************************)

createConstructorStubs[classIndex_Integer, ctors_List, useTypeChecking_] :=
	Module[{ctorsWithoutObjects, ctorsWithObjects, ctorList, ctorListForNonObjectDups,
			ctorListForObjectDups, indices, nonDups, dups, widest},
		(* Rely on the fact that argtype constants are negative numbers, and the only such numbers in each record. *)
		If[!FreeQ[ctors, x_Integer /; x <= TYPEBAD],
			(* TODO: Flesh out this error. Needs to report as problem in ctor. *)
			Message[LoadJavaClass::badtype];
			Return[]
		];
		(* Drop declaration; strip down to {index, argTypes...} *)
		ctorList = Delete[#, {2}]& /@ ctors;
		
		(* If not ctors defined, add the no-arg ctor. *)
		If[ctorList === {}, ctorList = {{0}}];
		
		ctorsWithoutObjects = Select[ctorList, FreeQ[#, TYPEOBJECT]&];
		ctorsWithObjects = Select[ctorList, !FreeQ[#, TYPEOBJECT]&];
		
		ctorListForNonObjectDups = Reverse /@ (ctorsWithoutObjects /. (x_Integer /; x < 0) :> argTypeToPattern[x]);
		ctorListForNonObjectDups = Split[Sort @ ctorListForNonObjectDups, Drop[#1, -1] === Drop[#2, -1] &];
		(* At this point, ctorListForNonObjectDups looks like:
		                           vv  Identical patterns are grouped  vv
			{{{argPat..., index}}, {{argPat..., index},{argPat..., index}}, {{argPat..., index}}}
		*)
		DebugPrint["ctorListForNonObjectDups = ", ctorListForNonObjectDups, Trigger:>$DebugDups];
		ctorListForNonObjectDups = Map[Last, ctorListForNonObjectDups, {2}];
		(* Now just {{index}, {index, index}, {index}, ...} *)
		
		nonDups = Select[ctorListForNonObjectDups, Length[#] == 1&];
		DebugPrint["nonDups = ", nonDups, Trigger:>$DebugDups];
		dups = Select[ctorListForNonObjectDups, Length[#] > 1&];
		DebugPrint["dups = ", dups, Trigger:>$DebugDups];
		(* The First below is just to strip outer list braces (the Select always returns one element) *)
		widest = pickWidest /@ (dups /. i_Integer :> First @ Select[ctorsWithoutObjects, First[#] == i&]);
		DebugPrint["widest = ", widest, Trigger:>$DebugDups];
		indices = First /@ Join[nonDups, widest];
		DebugPrint[indices, Trigger:>$DebugDups];
		(* These are the indices for which defs will be created (for ctors that don't have object args). *)
		(*indices = Last /@ Last /@ ctorListForNonObjectDups;*)
		Scan[createCtorDef[classIndex, useTypeChecking, {#}]&, Select[ctors, MemberQ[indices, First[#]]&]];
		
		ctorListForObjectDups = Reverse /@ (ctorsWithObjects /. (x_Integer /; x < 0) :> argTypeToPattern[x]);
		ctorListForObjectDups = Split[Sort @ ctorListForObjectDups, Drop[#1, -1] === Drop[#2, -1] &];
		(* At this point, ctorListForObjectDups looks like:
		                           vv  Identical patterns are grouped  vv
			{{{argPat..., index}}, {{argPat..., index},{argPat..., index}}, {{argPat..., index}}}
		*)
		indices = ctorListForObjectDups /. {__, n_Integer} :> n;
		(* indices looks like:  {{1}, {2,3,4}, {5}, {6}} *)
		(* The First below is just to strip outer list braces (the Select always returns one element) *)
		Scan[createCtorDef[classIndex, useTypeChecking, #]&, indices /. n_Integer :> First @ Select[ctors, First[#] == n &]];
	]

(* ctors looks like: {{index_Integer, declaration_String, paramTypes___Integer}...}.
    Only for object-containing dups will the list of ctors have > 1 element.
*)
createCtorDef[classIndex_Integer, useTypeChecking_, ctors_List] :=
	Module[{argTypes, argNames, argPats, indices, patternFunc},
		Assert @ MatchQ[ctors, {{_Integer, _String, ___Integer}..}];
		(* useTypeChecking == False means don't create Mathematica patterns for arg matching on LHS of definitions; just use _. *)
		patternFunc = If[useTypeChecking, argTypeToPattern, Blank[]&];
		If[Length[ctors] == 1,
			(* all cases except object-containing ctors with multiple identical arg sequences *)
			argTypes = Drop[First @ ctors, 2],
		(* else *)
			(* Object-containing ctors with multiple identical arg sequences. Need to create argTypes that has the
			    largest of all types in each slot.
			*)
			argTypes = Apply[Min, Drop[Transpose[ctors], 2], {1}]
		];
		argNames = Take[$argNames, Length[argTypes]];
		argPats = MapThread[Pattern, {argNames, patternFunc /@ argTypes}];
		indices = Sort[First /@ ctors];
		(javaNew[classIndex, Sequence @@ argPats] := javaConstructor[classIndex, ##])& @@
			Join[{indices}, createArgPairs[argTypes, argNames]];
	]

(* dups looks like {{index_Integer, paramTypes___Integer}, {index_Integer, paramTypes___Integer}, ...} *)
pickWidest[dups_List] :=
	Module[{sums},
		Assert @ MatchQ[dups, {{__Integer}..}];
		(* Logic here is to pick the version with largest args by using sum of type constants as measure.
		    Somewhat arbitrary, as type constants don't measure 'sizeof' accurately, but at least they are in proper order.
		*)
		sums = (Plus @@ Drop[#, 1])& /@ dups;
		dups[[ First @ Flatten @ Position[sums, Min[sums]] ]]
	]


(*********************************************  Method Handling  ************************************************)

createMethodStubs[classIndex_Integer, classContext_String, shortClassContext_String, meths_List, useTypeChecking_] :=
	Module[{methNames, methsWithoutObjects, methsWithObjects,
	        methListForNonObjectDups, methListForObjectDups, dups, nonDups, indices},
	        				(*  index,     decl,   isStatic,   name,   paramTypes *)
	    Assert[MatchQ[meths, {{_Integer, _String, True|False, _String, ___Integer}...}]];
		(* The following works because argtype constants are negative numbers, and the only such numbers in each record. *)
		If[!FreeQ[meths, x_Integer /; x <= TYPEBAD],
			(* TODO: Flesh out this error. Needs to report as problem in method. *)
			Message[LoadJavaClass::badtype];
			Return[]
		];

		methsWithoutObjects = Select[meths, FreeQ[#, TYPEOBJECT]&];
		methsWithObjects = Select[meths, !FreeQ[#, TYPEOBJECT]&];
		
		methListForNonObjectDups = Reverse /@ (methsWithoutObjects /. (x_Integer /; x < 0) :> argTypeToPattern[x]);
		methListForNonObjectDups = Split[Sort @ methListForNonObjectDups, Drop[#1, -3] === Drop[#2, -3] &];
		(* At this point, methListForNonObjectDups looks like:
		               vvvv  Identical name and patterns are grouped
			{{rec}, {rec, rec}, {rec}}
			where rec is {argPat..., name, decl, static, index}
		*)
		methListForNonObjectDups = Map[Last, methListForNonObjectDups, {2}];
		
		nonDups = Select[methListForNonObjectDups, Length[#] == 1&];
		DebugPrint["nonDups = ", nonDups, Trigger:>$DebugDups];
		dups = Select[methListForNonObjectDups, Length[#] > 1&];
		DebugPrint["dups = ", dups, Trigger:>$DebugDups];
		(* The First below is just to strip outer list braces (the Select always returns one element). Drop[#, {2,4}]
		    is to strip down to just index and paramTypes.
		*)
		widest = pickWidest /@ (dups /. i_Integer :> Drop[First @ Select[methsWithoutObjects, First[#] == i&], {2,4}]);
		DebugPrint["widest = ", widest, Trigger:>$DebugDups];
		(* These are the indices for which defs will be created (for meths that don't have object args). *)
		indices = First /@ Join[nonDups, widest];
		DebugPrint[indices, Trigger:>$DebugDups];

		Scan[createMethodDef[classIndex, classContext, shortClassContext, useTypeChecking, {#}]&,
			 Select[meths, MemberQ[indices, First[#]]&]
		];
		
		methListForObjectDups = Reverse /@ (methsWithObjects /. (x_Integer /; x < 0) :> argTypeToPattern[x]);
		methListForObjectDups = Split[Sort @ methListForObjectDups, Drop[#1, -3] === Drop[#2, -3] &];
		(* At this point, methListForObjectDups looks like:
		               vvvv  Identical name and patterns are grouped
			{{rec}, {rec, rec}, {rec}}
			where rec is {argPat..., name, decl, static, index}
		*)
		indices = methListForObjectDups /. {__, n_Integer} :> n;
		(* indices looks like:  {{1}, {2,3,4}, {5}, {6}} *)
		(* The First below is just to strip outer list braces (the Select always returns one element) *)
		Scan[createMethodDef[classIndex, classContext, shortClassContext, useTypeChecking, #]&,
			 indices /. n_Integer :> First @ Select[meths, First[#] == n &]
		]
	]


(* meths looks like:
	{{index_Integer, declaration_String, isStatic_True|False, name_String, paramTypes___Integer}..}
*)
createMethodDef[classIndex_Integer, ctxt_String, shortCtxt_String, useTypeChecking_, meths_List] :=
	Module[{methName, argTypes, argNames, argPats, indices, atLeastOneStatic, sym, patternFunc},
		Assert @ MatchQ[meths, {{_Integer, _String, True|False, _String, ___Integer}...}];
		(* useTypeChecking == False means don't create Mathematica patterns for arg matching on LHS of definitions; just use _. *)
		patternFunc = If[useTypeChecking, argTypeToPattern, Blank[]&];
		methName = toLegalName @ meths[[1, 4]];
		If[Length[meths] == 1,
			(* all cases except object-containing meths with multiple identical arg sequences *)
			argTypes = Drop[First @ meths, 4],
		(* else *)
			(* Object-containing meths with multiple identical arg sequences. Need to create argTypes that has the
			    largest of all types in each slot.
			*)
			argTypes = Apply[Min, Drop[Transpose[meths], 4], {1}]
		];
		argNames = Take[$argNames, Length[argTypes]];
		argPats = MapThread[Pattern, {argNames, patternFunc /@ argTypes}];
		indices = Sort[First /@ meths];
		atLeastOneStatic = Or @@ (#[[3]]& /@ meths);
		With[{indices = indices, argc = Length[argTypes], argPairs = createArgPairs[argTypes, argNames]},
			(* The instance head is just to prevent mistaken matching with an integer
			   in arg sequence of a static function. *)
			ToExpression[ctxt <> "JPrivate`" <> methName][instance[sym_], Sequence @@ argPats] :=
					callJava[{classIndex, 2, sym, indices, 1}, argc, Sequence @@ argPairs];
			If[atLeastOneStatic,
				With[{sym = Symbol[ctxt <> methName]},
					sym[Sequence @@ argPats] := callJava[{classIndex, 2, Null, indices, 1}, argc, Sequence @@ argPairs];
					sym[args___] := (
						Switch[Length[{args}],
							0,
								Message[Java::argxs0, HoldForm[sym]],
							1,
								Message[Java::argxs1, HoldForm[sym], args],
							_,
								Message[Java::argxs, HoldForm[sym], {args}]
						];
						$Failed);
					(* Downvalues of isJavaStaticSymbol are used to record which symbols in a context hav been given defs.
					   This is used only in clearOutClassContext, to avoid clearing non-Java symbols in case the same
					   context name is being used by a Mathematica package. No need to do this for the shortCtxt symbols,
					   as they do not need to be cleared when the class is unloaded. They just point to their deep-context
					   counterparts, which will get cleared.
					*)
					isJavaStaticSymbol[ctxt <> methName] = True;
					isJavaStaticSymbol[shortCtxt <> methName] = True;
					(* Now make def also available in "short" class context. Evaluate is necessary here but not
					   for DownValues above due to strangeness of Set partial lhs eval. If user has specified to not allow
					   short contexts, the short context will be the same as the long one, hence the test.
					*)
					If[shortCtxt != ctxt, Evaluate[ToExpression[shortCtxt <> methName]] = sym]
				]
			]
		];
	]


(******************************************  Field Handling  *************************************************)

Java::setfield = "Trying to set field to illegal value."

(* fields is:  {{index_Integer, isStatic_True|False, type_String, name_String, type_Integer}...}
*)
createFieldStubs[classIndex_Integer, classContext_String, shortClassContext_String, fields_List, useTypeChecking_] :=
	Module[{fieldNames},
		Assert @ MatchQ[fields, {{_Integer, True|False, _String, _String, _Integer}...}];
		(* Bit of a hack follows; rely on the fact that argtype constants are negative numbers,
		   and the only such numbers in each record.
		*)
		If[!FreeQ[fields, x_Integer /; x <= TYPEBAD],
			(* Flesh out this error. Needs to report as problem in field. *)
			Message[LoadJavaClass::badtype];
			Return[]
		];
		Scan[createFieldDef[classIndex, classContext, shortClassContext, useTypeChecking, #]&, fields];
	]


(* field looks like:
	{index_Integer, isStatic_True|False, type_String, name_String, type_Integer}
*)
createFieldDef[classIndex_Integer, ctxt_String, shortCtxt_String, useTypeChecking_, field_List] :=
	Module[{fieldName, index, argName, argType, patternFunc},
		Assert @ MatchQ[field, {_Integer, True|False, _String, _String, _Integer}];
		(* useTypeChecking == False means don't create Mathematica patterns for arg matching on LHS of definitions; just use _. *)
		patternFunc = If[useTypeChecking, argTypeToPattern, Blank[]&];
		fieldName = toLegalName @ field[[4]];
		index = field[[1]];
		argName = First[$argNames];
		argType = field[[5]];
		With[{index = index, privSym = ToExpression[ctxt <> "JPrivate`" <> fieldName],
				argName = argName, argType = argType, pat = Pattern @@ {argName, patternFunc[argType]}},
			(* The Private` symbol fieldTag is just a placeholder to distinguish field accesses from
			   no-arg method calls of the same name.
			*)
			privSym[fieldTag, instance_Symbol] := callJava[{classIndex, 3, instance, {index}, 1}, 0];
			privSym[fieldTag, instance_Symbol, pat] := callJava[{classIndex, 3, instance, {index}, 1}, 1, argType, argName];
			If[field[[2]] === True,
		 		(* is static *)
		 		(* Note there is a very small problem with my scheme for accessing fields in the static case.
		 		   I want the user to be able to get the value with java`awt`Button`foo, but that requires
		 		   assigning an ownvalue to java`awt`Button`foo, which will cause a problem with the func
		 		   def for that symbol that would be created if a static method had the same name as a static field.
		 		   Bail on this for now. Maybe it's OK to document some hack for this unlikely circumstance...
		 		   Note it's only a problem for getting, not setting, and you could avoid even that by using
		 		   the object syntax for calling a static method.
		 		*)
		 		makeStaticFieldDefs[ToHeldExpression[ctxt <> fieldName], ctxt <> fieldName, classIndex, index, argType, pat];
		 		If[shortCtxt != ctxt,
		 			(* Because we set UpValues for Set calls, it is not enough to just define the shortCtxt field symbols to be the
		 			   deep context symbols, as is done with methods. Instead, we must explicitly make definitions for the shortCtxt ones.
		 			*)
		 			makeStaticFieldDefs[ToHeldExpression[shortCtxt <> fieldName], shortCtxt <> fieldName, classIndex, index, argType, pat]
		 		]
			]
		];
	]
	
	makeStaticFieldDefs[Hold[sym_], symStr_, classIndex_, index_, argType_, pat_] := 
		(* The !ValueQ test prevents this from being called twice on a symbol. This will happen if you load two classes
		   with the same short context. Calling it twice can cause all sorts of bad behavior.
		*)
		If[!ValueQ[sym],
			(* For consistency we route everything through setField for statics even though Set method has no drawbacks
			   for statics. If I wanted to completely remove all reliance on Set for statics I could just remove the
			   following one line. There is no reason to want to do that, though.
			*)
			sym /: Set[sym, val_] := setField[sym, val];
			sym /: setField[sym, val_] :=
						If[MatchQ[val, pat],
							callJava[{classIndex, 3, Null, {index}, 1}, 1, argType, val];
							val,
						(* else *)
							Message[Java::fldxs, HoldForm[sym], val];
							$Failed
						];
			(* Must make this def last. *)
			sym := callJava[{classIndex, 3, Null, {index}, 1}, 0];
			(* Downvalues of isJavaStaticSymbol are used to record which symbols in a context have been given defs.
				This is used only in clearOutClassContext, to avoid clearing non-Java symbols in case the same
				context name is being used by a Mathematica package.
			*)
			isJavaStaticSymbol[symStr] = True;
		]


(*****************************************   Returning Refs   **********************************************)

(* Note that ReturnAsJavaObject sets up an "environment" where all calls return by ref. This means that
   ReturnAsJavaObject[foo[obj@method[]]] will work, but be careful if there are more Java calls embedded in the
   expression, as in ReturnAsJavaObject[obj@method[SomeClass`FOO]], as these deeper calls will also return by ref.
*)

ByRef = ReturnAsJavaObject   (* ByRef is deprecated. *)

SetAttributes[ReturnAsJavaObject, HoldAll]

ReturnAsJavaObject[x_] := Block[{$byRef = True}, x]


Val = JavaObjectToExpression   (* Val is deprecated. *)

JavaObjectToExpression[x_?JavaObjectQ] := jVal[x]

JavaObjectToExpression[x_] := x   (* Perhaps this should issue a message? *)


(******************************************  ExternalCall fix  *********************************************)

(* Calls to external functions via the standard Install mechanism are accomplished by the function ExternalCall.
   Unfortunately, ExternalCall is deficient in its handling of aborts. It also needs to dynamically choose
   the link to use to communicate with Java. Thus, I need my own version, jlinkExternalCall.
   I also need my own version of DefineExternal to create definitions that call jlinkExternalCall instead of
   ExternalCall. 
   
   These functions would have to be revised if anything about the internals of Mathematica's Install/Uninstall
   mechanism changed.
*)

(* This is similar to DefineExternal except that it creates defs that call jlinkExternalCall instead of ExternalCall.
   This function is called directly from Java.
*)
jlinkDefineExternal[p_String, a_, n_] := 
	Module[{e, pat = ToHeldExpression[p], args = ToHeldExpression[a]}, 
		If[pat === $Failed || args === $Failed, 
			Message[DefineExternal::des, n, InputForm[$CurrentLink]],
		(* else *)
			e = Hold[_ := jlinkExternalCall[_, CallPacket[_, _]]];
			e = ReplaceHeldPart[e, pat, {1, 1}];
			e = ReplaceHeldPart[e, Hold[getActiveJavaLink[]], {1, 2, 1}];
			e = ReplacePart[e, n, {1, 2, 2, 1}];
			e = ReplaceHeldPart[e, args, {1, 2, 2, 2}];
			ReleaseHold[e];
			System`Dump`defined = Append[System`Dump`defined, HoldForm @@ pat /.
				{Literal[ThisLink] -> $CurrentLink, Literal[$CurrentLink] -> $CurrentLink}]
		];
	]

(* This function differs functionally from ExternalCall in that it wraps the write-read pair in AbortProtect, so that you
   cannot leave the link in an "off-by-one" state by aborting between the write and read. This AbortProtect does not
   prevent the necessary behavior that user aborts fired while the kernel is blocking in LinkRead are sent to Java
   as MLAbortMessages. In other words, the ability to abort Java computations is not affected. The AbortProtect does
   prevent the behavior of being able to do a "hard" abort via the two-step combination of
   "Interrupt Evaluation/Abort Command Being Evaluated". This procedure causes Mathematica to treat the
   abort like any other abort and ignore that it is in LinkRead. This is not very useful, though, since the link will
   probably be out of sync because the result is not read. The correct way to handle this is to select "Kill linked program"
   in the Interrupt dialog box, not "Abort Command Being Evaluated". This causes Java to quit.
   
   The code itself is quite different from ExternalCall. Gone is the need for ExternalAnswer and the silly recursive way
   in which that was implemented.
   
   The link that will be passed in here is the one given by getActiveJavaLink[].
   
   This function also rejects preemptive calls into Java when they are unsafe, which is the case
   when they are preemptive, reentrant (call into Java is already occurring), but not originating on the UI link.
   
   TODO: Code for AbortProtect suggests that it is not interruptible by preemptive evals. Doesn't that
   mean that (at least until that gets changed) there is no need for all the reentrancy control in this func?
*)

jlinkExternalCall[link_LinkObject, packet_CallPacket] :=
	Block[{$CurrentLink = link, pkt = packet, res, isPreemptive = TrueQ[MathLink`IsPreemptive[]],
				$externalCallLink = $externalCallLink, $inPreemptiveCallToJava = $inPreemptiveCallToJava},
		AbortProtect[
			(* Reject as unsafe calls that are: preemptive, reentrant on the same link,
			   and not just callbacks from a preemptive comp that began in Java or callbacks
			   during a preemptive comp that went out to Java.
			*)
			If[isPreemptive && $externalCallLink === link && 
						!$inPreemptiveCallFromJava && !TrueQ[$inPreemptiveCallToJava],
				Message[Java::preemptive];
				$Failed,
			(* else *)
				$externalCallLink = link;
				If[isPreemptive, $inPreemptiveCallToJava = True];
				While[True,
					If[LinkWrite[link, pkt] === $Failed, Return[$Failed]];
					If[link === $InternalLink, Java`DispatchToJava[link]];
					res = LinkReadHeld[link];
					Switch[res,
						Hold[EvaluatePacket[_]],
							(* Re-enable aborts during the computation in Mathematica of EvaluatePacket contents, but have
							   them just cause $Aborted to be returned to Java, not call Abort[].
							*)
							pkt = ReturnPacket[CheckAbort[res[[1,1]], $Aborted]],
						Hold[ReturnPacket[_]],
							Return[res[[1,1]]],
						Hold[_],
							Return[res[[1]]],
						_,
							Return[res]
					]
				]
			]	
		]
	]


(* This gives the link that will be used for any given call to Java. Note that preemptive calls
   to Java will never use the UI link unless they are just callbacks during a preemptive comp
   that began in Java. Note also that JavaUILink[] will never be returned unless it is safe to use it.
*)
getActiveJavaLink[] :=
	Block[{isPreemptive = TrueQ[MathLink`IsPreemptive[]]},
		If[!isPreemptive && $ParentLink === JavaUILink[] && $ParentLink =!= Null ||
				isPreemptive && $inPreemptiveCallFromJava,
			JavaUILink[],
		(* else *)
			JavaLink[]
		]
	]
	
	
Java::preemptive = "Calls into Java cannot be made from a preemptive computation while another call into Java is in progress."


(******************************************  Creating instance defs  *********************************************)

(* Called only from Java, for returning objects to Mathematica whose classes have not been loaded by user. *)
loadClassAndCreateInstanceDefs[clsName_String, obj_Symbol] :=
	Module[{cls},
		cls = loadClassFromJava[clsName, obj];
		If[Head[cls] === JavaClass,
			createInstanceDefs[classIndexFromClass[cls], obj],
		(* else *)
			$Failed
		]
	]


(* Called from Java whenever classes need to be loaded by Java code. This is currently in two circumstances: loading
   parent classes of a class the user has manually loaded using LoadJavaClass or JavaNew; or classes loaded because an object
   of their type is being returned from Java.
   Note that if you want to have a class loaded with your own settings for the options of LoadJavaClass, then you had better
   load it yourself, before it is autoloaded for you.

NOTE: I originally had AllowShortContext->False here, but have decided to remove it. Perhaps put it back when I implement
my TODO of having later calls to LoadJavaClass with higher context visibility have the requested effect.
*)
loadClassFromJava[clsName_String, obj_Symbol] := LoadJavaClass[clsName, obj, StaticsVisible->False]
loadClassFromJava[Null, _] = Null  (* For superclass of java.lang.Object. *)


(* These have just the following definitions. Defs are never added (instead, upvalues are placed on the JavaObjectN symbols). *)
JavaObjectQ[_] = False
JavaObjectQ[Null] = True
classIndexFromInstance[_] = $Failed


(* This function cannot be changed without also changing unloadClass. *)

createInstanceDefs[classIndex_Integer, obj_Symbol] :=
	Module[{clsName, arrayDepth = 0, arrayType, nameLen, complexClass},
		SetAttributes[obj, {HoldAllComplete}];
		JavaObjectQ[obj] ^= True;
		classIndexFromInstance[obj] ^= classIndex;
		addToJavaBlock[obj];
		clsName = classNameFromIndex[classIndex];
		Assert[clsName =!= $Failed];
		nameLen = StringLength[clsName];
		arrayDepth = Which[nameLen >= 3 && StringTake[clsName, 3] === "[[[",
							 3,
						   nameLen >= 2 && StringTake[clsName, 2] === "[[",
							 2,
						   nameLen >= 1 && StringTake[clsName, 1] === "[",
						     1,
						   True,
						     0
					 ];
		arrayType = If[arrayDepth === 0, Null, StringTake[clsName, {arrayDepth + 1}]];
		(* This defeats normal precedence for @ operator. Needed for chaining: obj@meth1[]@meth2[]. *)
		obj[(meth:_[___])[args___]] := obj[meth][args];
		obj[meth_[args___]] := javaMethod[obj, meth, args];
		obj[field_Symbol] := javaField[obj, field];
		obj /: setField[obj[field_Symbol], val_] := If[# === $Failed, $Failed, val]& [javaField[obj, field, val]];
		If[arrayType =!= Null,
			Switch[arrayType,
				"B" | "C" | "S" | "I" | "J",
					obj /: isJavaIntegerArray[obj, arrayDepth] = True,
				"F" | "D",
					obj /: isJavaRealArray[obj, arrayDepth] = True,
				"Z",
					obj /: isJavaBooleanArray[obj, arrayDepth] = True,
				"L",
					obj /: isJavaObjectArray[obj, arrayDepth] = True;
					Which[StringMatchQ[clsName, "*java.lang.String*"],
							 obj /: isJavaStringArray[obj, arrayDepth] = True,
						  StringMatchQ[clsName, "*java.math.BigDecimal*"],
							 obj /: isJavaBigDecimalArray[obj, arrayDepth] = True,
						  StringMatchQ[clsName, "*java.math.BigInteger*"],
							 obj /: isJavaBigIntegerArray[obj, arrayDepth] = True,
						  StringMatchQ[clsName, "*com.wolfram.jlink.Expr*"],
							 obj /: isJavaExprArray[obj, arrayDepth] = True,
						  complexClassName = classNameFromClass[GetComplexClass[]];
						  StringQ[complexClassName] && StringMatchQ[clsName, "*" <> complexClassName <> "*"],
							 obj /: isJavaComplexArray[obj, arrayDepth] = True
					]
			]
		];
		Format[obj, OutputForm] = Format[obj, TextForm] = "<<JavaObject[" <> clsName <> "]>>";
		(obj /: MakeBoxes[obj, fmt_] = InterpretationBox[RowBox[{"\[LeftGuillemet]", RowBox[{"JavaObject", "[", #, "]"}], "\[RightGuillemet]"}], obj])& [clsName];
		obj
	]


(* Called during Un/InstallJava to wipe out defs created as classes are loaded and objects created. *)
clearJavaDefs[] :=
	(
		clearAllClassContexts[];
		Clear[isJavaStaticSymbol];
		Clear[javaNew];
		destroyLoadedClassInfo[];  (* Wipes out stored JavaClass records. *)
		ClearAll["JLink`Objects`*"];
		Clear[$instanceFuncCache];
	)


(****************************************  Method call-building utils  ************************************)

(* This creates parameter sequence on rhs of ctor/method defs. *)
createArgPairs[argTypes_, argNames_] := Flatten[Transpose[{argTypes, argNames}]]

(* This creates the patterns on the lhs of definitions. *)
argTypeToPattern[n_Integer] :=
	Switch[n,
		TYPEBOOLEAN,
			Return[True | False],
		TYPEBYTE | TYPECHAR | TYPESHORT | TYPEINT | TYPELONG,
			Return[_Integer],
		TYPEFLOAT | TYPEDOUBLE,
			Return[_Real],
		TYPECOMPLEX,
			Return[_?isComplex],
		TYPESTRING,
			Return[_?isString],
		TYPEEXPR,
			Return[_],
		TYPEOBJECT,
			Return[_?JavaObjectQ],
		TYPEBIGINTEGER,
			Return[_?isBigInteger],
		TYPEBIGDECIMAL,
			Return[_?isBigDecimal],
		TYPEDOUBLEORINT | TYPEFLOATORINT,
			Return[_?NumberQ],
		TYPEBOOLEAN + TYPEARRAY1,
			Return[_?isTrueFalseList],
		TYPEBYTE + TYPEARRAY1 | TYPECHAR + TYPEARRAY1 | TYPESHORT + TYPEARRAY1 |
				TYPEINT + TYPEARRAY1 | TYPELONG + TYPEARRAY1,
			Return[_?isIntegerList],
		TYPEFLOAT + TYPEARRAY1 | TYPEDOUBLE + TYPEARRAY1,
			Return[_?isRealList],
		TYPECOMPLEX + TYPEARRAY1,
			Return[_?isComplexList],
		TYPESTRING + TYPEARRAY1,
			Return[_?isStringList],
		TYPEOBJECT + TYPEARRAY1,
			Return[_?isObjectList],
		TYPEBIGINTEGER + TYPEARRAY1,
			Return[_?isBigIntegerList],
		TYPEBIGDECIMAL + TYPEARRAY1,
			Return[_?isBigDecimalList],
		TYPEEXPR + TYPEARRAY1,
			Return[_?isExprList],
		TYPEDOUBLEORINT + TYPEARRAY1 | TYPEFLOATORINT + TYPEARRAY1,
			Return[_?isNumberList],
		TYPEBOOLEAN + TYPEARRAY2,
			Return[_?isTrueFalseArray2],
		TYPEBYTE + TYPEARRAY2 | TYPECHAR + TYPEARRAY2 | TYPESHORT + TYPEARRAY2 |
				TYPEINT + TYPEARRAY2 | TYPELONG + TYPEARRAY2,
			Return[_?isIntegerArray2],
		TYPEFLOAT + TYPEARRAY2 | TYPEDOUBLE + TYPEARRAY2,
			Return[_?isRealArray2],
		TYPECOMPLEX + TYPEARRAY2,
			Return[_?isComplexArray2],
		TYPESTRING + TYPEARRAY2,
			Return[_?isStringArray2],
		TYPEDOUBLEORINT + TYPEARRAY2 | TYPEFLOATORINT + TYPEARRAY2,
			Return[_?isNumberArray2],
		TYPEOBJECT + TYPEARRAY2,
			Return[_?isObjectArray2],
		TYPEBIGINTEGER + TYPEARRAY2,
			Return[_?isBigIntegerArray2],
		TYPEBIGDECIMAL + TYPEARRAY2,
			Return[_?isBigDecimalArray2],
		TYPEEXPR + TYPEARRAY2,
			Return[_?isExprArray2],
		TYPEBOOLEAN + TYPEARRAY3,
			Return[_?isTrueFalseArray3],
		TYPEBYTE + TYPEARRAY3 | TYPECHAR + TYPEARRAY3 | TYPESHORT + TYPEARRAY3 |
				TYPEINT + TYPEARRAY3 | TYPELONG + TYPEARRAY3,
			Return[_?isIntegerArray3],
		TYPEFLOAT + TYPEARRAY3 | TYPEDOUBLE + TYPEARRAY3,
			Return[_?isRealArray3],
		TYPECOMPLEX + TYPEARRAY3,
			Return[_?isComplexArray3],
		TYPESTRING + TYPEARRAY3,
			Return[_?isStringArray3],
		TYPEDOUBLEORINT + TYPEARRAY3 | TYPEFLOATORINT + TYPEARRAY3,
			Return[_?isNumberArray3],
		TYPEOBJECT + TYPEARRAY3,
			Return[_?isObjectArray3],
		TYPEBIGINTEGER + TYPEARRAY3,
			Return[_?isBigIntegerArray3],
		TYPEBIGDECIMAL + TYPEARRAY3,
			Return[_?isBigDecimalArray3],
		TYPEEXPR + TYPEARRAY3,
			Return[_?isExprArray3],
		_,
			_List   (* For 4-deep and deeper arrays, just use a minimal pattern. *)
	]

(* Equivs looks like:
     when called for constructor:  {{argPat..., declaration, index}..}
     when called for method:       {{argPat..., name, decl, static, index}..}
*)
warnForDups[class_String, equivs_List] :=
	Module[{name, decs},
		If[Length[equivs] > 1,
			If[Head @ equivs[[1,-2]] === String,
				(* is a constructor *)
				decs = #[[-2]]& /@ equivs;
				Message[LoadJavaClass::ambigctor, class, decs],
			(* else *)
				(* is a method *)
				name = equivs[[1,-4]];
				decs = #[[-3]]& /@ equivs;
				Message[LoadJavaClass::ambig, class, name, decs]
			]
		]
	]

toLegalName[s_String] := StringReplace[s, "_" -> "U"]


(* As an optimization, create ahead of time a list of arg symbols used in Java function definitions. This is a surprisingly
   expensive operation to do repeatedly. Note the arbitrary max of 40 args per method.
*)
$argNames = Table[ToExpression["arg$" <> ToString[n]], {n, 1, 40}];



(******************************************  Class-related utils  *****************************************)

(*
	There are three ways to refer to a Java class after it has been loaded:
	
	(1) By its fully-qualified name as a string  "com.foo.MyClass"
	(2) By its class index, which is an index into the vector of ClassRecords held in Java.
		This is the only way classes are known to Java. You must always convert to this index
		when crossing over into Java. This method is not visible to users.
	(3) By a JavaClass expr, which is what it returned by LoadJavaClass. A JavaClass just encapsulates
		the string form with the index form: JavaClass["com.foo.MyClass", 17]
		
	Want users to be able to refer to classes as strings or as JavaClass exprs (index form is
	not documented), although JavaClass is the preferred way. We have functions to convert among
	these representations.
*)


JavaClass::notfound = "No class named `1` has been loaded."

(***** Next set are the only functions that know anything about the structure of JavaClass:
          JavaClass[name_String, index_Integer]
*****)

createJavaClass[name_String, index_Integer] := JavaClass[name, index]

classNameFromClass[cls_JavaClass] := cls[[1]]
classIndexFromClass[cls_JavaClass] := cls[[2]]


(* TODO: Need to examine error outcomes and reporting in the next set of functions... *)

classIndexFromName[cls_String] := classIndexFromClass[classFromName[cls]]

classNameFromIndex[i_Integer] := classNameFromClass[classFromIndex[i]]

(*********
		Next set are the only funcs that know anything about how class info is stored in Mathematica
		(i.e., as DownValues of loadedClasses).
*********)

classFromName[cls_String] :=
	Module[{classes, legalClassName},
		legalClassName = toLegalName[cls];
		classes = Select[DownValues[loadedClasses], classNameFromClass[#[[2, 1]]] === legalClassName &];
		If[classes === {},
			$Failed,
		(* else *)
			Assert[Length[classes] === 1];
			First[classes] [[2, 1]]
		]
	]

(* Need this function to detect if a user is using a JavaClass expression that refers to an unloaded class.
   This could happen if the Java runtime is quit and restarted.
*)
isLoadedClass[jc_JavaClass] := ValueQ[loadedClasses[classIndexFromClass[jc]]]

classFromIndex[i_Integer] := loadedClasses[i] [[1]]

parentClassIndexFromClassIndex[i_Integer] := loadedClasses[i] [[2]]

contextNameFromClassIndex[i_Integer] := loadedClasses[i] [[3]]

(* The only reason we store the "is interface" flag is so that it can be retrieved with this function
   so as to issue a special message in JavaNew.
*)
interfaceQFromIndex[i_Integer] := loadedClasses[i] [[4]]

usesShortContext[i_Integer] := loadedClasses[i] [[5]]

storeClass[i_Integer, cls_JavaClass, parentClsIndex:(Null | _Integer), isInterface:(True | False), usesShortContext:(True | False)] :=
	loadedClasses[i] = {cls, parentClsIndex, toContextName[toLegalName[classNameFromClass[cls]]], isInterface, usesShortContext}

allLoadedClasses[] := #[[2, 1]]& /@ DownValues[loadedClasses]

destroyLoadedClassInfo[] := Clear[loadedClasses]

removeClass[i_Integer] := loadedClasses[i] =.


(* TODO: These should Block some symbols that would be useful (e.g. $ClassContext). *)

callOnLoadClassMethod[i_Integer] := jOnLoadClass[i]
callOnUnloadClassMethod[i_Integer] := jOnUnloadClass[i]

callAllUnloadClassMethods[] := callOnUnloadClassMethod /@ classIndexFromClass /@ allLoadedClasses[]

clearAllClassContexts[] := Scan[clearOutClassContext, allLoadedClasses[]];


(* This one is used only when Java is being started/stopped. *)
clearOutClassContext[jc_JavaClass] := 
	Module[{ctxt = contextNameFromClassIndex[classIndexFromClass[jc]], shortCtxt, nms, javaNames},
		Assert[Head[ctxt] === String];
		(* Must get rid of $ContextPath or symbols in visible ccontexts will have their names returned without
		   the context prefix.
		*)
		nms = Block[{$ContextPath}, Names[ctxt <> "*"]];
		(* Downvalues of isJavaStaticSymbol are used to record which symbols in a context have been given defs.
		   We use this to avoid clearing non-Java symbols in case the same context name is being used by a
		   Mathematica package. We only need be concerned with statics, since non-statics are only created
		   in the special JPrivate` context, which we can be sure will have no conflicts.
		*)
		javaNames = Select[nms, isJavaStaticSymbol];
		(* ClearAll["sym1", "sym2", ...] is vastly more expensive than ClearAll[sym1, sym2, ...] or ClearAll["ctxt`*"],
		   so avoid the first method at all costs. We only do selective clearing when it is necessary (this will likely
		   be only in cases where a read-in context has the same name as a Java-created one), and when we do, do it
		   for symbol names rather than strings.
		*)
		If[javaNames =!= {},
			If[Length[nms] == Length[javaNames],
				ClearAll @@ {ctxt <> "*"},
			(* else *)
				(* The Unevaluated wrapped around each symbol does not interfere with ClearAll. *)
				ClearAll @@ (ToExpression[#, InputForm, Unevaluated]& /@ javaNames)
			]
		];
		If[Names[#] =!= {}, ClearAll[#]]& [ctxt <> "JPrivate`*"];
		If[usesShortContext[classIndexFromClass[jc]],
			shortCtxt = shortClassContextFromClassContext[ctxt];
			nms = Block[{$ContextPath}, Names[shortCtxt <> "*"]];
			If[shortCtxt == "System`",
				(* The trick above with $ContextPath doesn't work for the System` context. We need to explicitly force the
				   name strings to begin with "System`", as that's how they are recorded in isJavaStaticSymbol.
				*)
				nms = ("System`" <> #)& /@ nms
			];
			javaNames = Select[nms, isJavaStaticSymbol];
			If[javaNames =!= {},
				If[Length[nms] == Length[javaNames],
					ClearAll @@ {shortCtxt <> "*"},
				(* else *)
					ClearAll @@ (ToExpression[#, InputForm, Unevaluated]& /@ javaNames)
				]
			];
		]
	]


shortClassContextFromClassContext[ctxt_String] :=
	StringDrop[ctxt, If[Length[#] > 1, #[[-2]], 0]]& @ Union[Flatten[StringPosition[ctxt, "`"]]]

toContextName[clsName_String] :=
	If[StringTake[clsName, 1] === "[", "java`lang`Array`", StringReplace[clsName, "." -> "`"] <> "`"]


End[]