(* :Title: Exceptions.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 3.1 *)

(* :Mathematica Version: 4.0 *)
		     
(* :Copyright: J/Link source code (c) 1999-2005, Wolfram Research, Inc. All rights reserved.

   Use is governed by the terms of the J/Link license agreement, which can be found at
   www.wolfram.com/solutions/mathlink/jlink.
*)

(* :Discussion:
   Handling of Java exceptions arising from calls from M into Java. Some of these functions
   are called directly from Java.
	
   This file is a component of the J/Link Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   J/Link uses a special system wherein one package context (JLink`) has its implementation
   split among a number of .m files. Each component file has its own private context, and also
   potentially introduces public symbols (in the JLink` context) and so-called "package" symbols,
   where the term "package" comes from Java terminology, referring to symbols that are visible
   everywhere within the implementation of J/Link, but not to clients. The JLink.m file itself
   is produced by an automated tool from the component files and contains only declarations.
   
   Do not modify the special comment markers that delimit Public- and Package-level exports.
*)



(*<!--Public From Exceptions.m

GetJavaException::usage =
"GetJavaException[] returns the Java exception object that was thrown in the most recent call from Mathematica to Java. It returns Null if no exception was thrown in the most recent call. You can use GetJavaException in conjunction with $JavaExceptionHandler to implement a custom exception-handling scheme in Mathematica."

$JavaExceptionHandler::usage =
"$JavaExceptionHandler allows you to control how exceptions thrown in Java are handled in Mathematica. The default behavior is for exceptions to appear as messages in Mathematica. If you want to override this behavior (for example to temporarily silence messages from exceptions), assign a value to $JavaExceptionHandler. The value of $JavaExceptionHandler is treated as a function that will be passed 3 arguments: the symbol associated with the message (this will currently always be the symbol Java), the message tag (this will be the string \"excptn\" for a typical exception or \"pexcptn\" for an exception generated by a \"manual return\" method where the exception occurs after the method has manually sent its result back to Mathematica), and the descriptive string of text associated with the message. You will typically set $JavaExceptionHandler within a Block so that its effect will be limited to a precisely defined segment of code, as in the following example that silences messages: Block[{$JavaExceptionHandler = Null&}, obj@method[]]. You can use GetJavaException[] within your handler function to obtain the actual Java exception object that was thrown."

-->*)

(*<!--Package From Exceptions.m

// All these are called only from Java
prepareForManualReturn
handleCleanException
autoException
manualException

-->*)


(* Current context will be JLink`. *)

Begin["`Exceptions`Private`"]


GetJavaException[] := jGetException[]


(* The Java::excptn message is called directly from Java in one circumstance, so don't change it. *)
Java::excptn = "A Java exception occurred: `1`."
Java::pexcptn = "A Java exception occurred after the result was returned to Mathematica: `1`."

(* This section keeps the link as an arg that is passed around, so it is forward-looking to the possibility
    of more than one VM running at a time.
*)

(* Exception handling has to accommodate manual/non-manual functions, and for manuals, whether the exception
    is thrown before, during, or after the user has sent a complete result on the link. The basic idea is that
    if a function has a manual return, in most cases Mathematica must read twice from the link: the first time
    to get the answer, and again to get exception info. The only exception (no pun intended) is where
    the exception occurs before anything has been put on the link--in this case there is only one read, which
    gets the exception info.

    Several of the functions here are called directly from Java. they are:
		prepareForManualReturn
		handleCleanException
	These next two are just wrappers to identify whether exc is from a manual func or not. All exception info is returned
	wrapped in one of these two wrappers.
    	manualException	   exception thrown after a function declares itself manual.
    	autoexception	   exception thrown in either: non-manual function, manual before it is declared, or other Java
    					   functions than invoking user code (e.g., LoadClass, ReleaseObject, etc.)
*)

(* Called from Java when ml.beginManual() is executed. Basically, this function bails us out of the default
    ExternalAnswer loop and into our own equivalent, readManualResult. Once we're in readManualResult, we know
    we're in "manual mode" and can interpret things properly.
*)
prepareForManualReturn[link_LinkObject] :=
	(
		DebugPrint["in prepareForManualReturn", Trigger :> $DebugExceptions];
		readManualResult[link]
	)

readManualResult[link_LinkObject] :=
	Module[{res},
		DebugPrint["in readManualResult, about to call LinkReadHeld", Trigger :> $DebugExceptions];
		(* Shutting off $Messages here is so that the LinkRead::linkep message ("unexpected end of packet",
		   new in 4.0) will not show up if an exception occurs in the middle of sending the result. This message
		   is technically correct, but unnecessarily disturbing since this is part of the plan. But I do want the
		   LinkRead::linkd (link died) message to appear, so this is handled below in the fall-through Switch branch.
		*)
		Block[{$Messages}, res = LinkReadHeld[link]];
		DebugPrint["in readManualResult, back from LinkReadHeld. res was: ", res, Trigger :> $DebugExceptions];
		Switch[res,
			Hold[$Aborted],
				(* Either an exc was thrown during the interval where the result was being put on the link,
				    or the func deliberately returned $Aborted as the result. Either case is handled correctly.
				*)
				DebugPrint["in readManualResult, got $Aborted", Trigger :> $DebugExceptions];
				handleDirtyException @ readException @ link,
			Hold[_EvaluatePacket],
				DebugPrint["in readManualResult, got EvaluatePacket", Trigger :> $DebugExceptions];
				If[LinkWrite[link, ReturnPacket[CheckAbort[res[[1,1]], $Aborted]]] === $Failed,
					$Failed,
				(* else *)
					readManualResult[link]
				],
			Hold[_prepareForManualReturn],
				(* Multiple calls to beginManual(). Just ignore it and call readManualResult again. *)
				DebugPrint["in readManualResult, got prepareForManualReturn", Trigger :> $DebugExceptions];
				readManualResult[link],
			Hold[_manualException],
				(* Was a manual return, but nothing was put on link by user. Just call handleCleanException. *)
				DebugPrint["in readManualResult, got manualException", Trigger :> $DebugExceptions];
				handleCleanException @@ res,   (* Apply just gets rid of the Hold. *)
			Hold[_ReturnPacket],
				(* Normal return, but user wrapped it in ReturnPacket. *)
				handlePostException @ readException @ link;
				res[[1,1]],
			_Hold,
				(* Normal return. *)
				DebugPrint["in readManualResult, got normal return", Trigger :> $DebugExceptions];
				handlePostException @ readException @ link;
				res[[1]],
			_,
				(* I think it will always be $Failed, and that link is dead. *)
				Assert[res === $Failed && !MemberQ[Links[], link]];
				If[!MemberQ[Links[], link],
					Message[LinkObject::linkd, ToString[link]],
				(* else *)
					(* Don't think this will ever be taken, but if link is still alive there is presumably
					   exception info remaining to be read.
					*)
					readException[link]
				];
				res
		]
	]

(* This performs the "second read" for exception info. This will either be Null for no exception, or manualException
    or autoException if there was an exception. The result of readException is fed into a "handle" function that knows
    whether it's a clean, dirty, or post exception.
*)
readException[link_LinkObject] := LinkRead[link]

(* Dirty exceptions are those that are thrown while the user is in the middle of sending a result back. *)
handleDirtyException[e_manualException] := 
	(
		reportException[e];
		$Failed
	)
	
(* This def handles the case of a manual function that deliberately returns $Aborted (no exceptions are thrown).
    In such a circumstance, readException pulls off the normal Null second return, and we want to ignore exception
    issues entirely.
*)
handleDirtyException[Null] = $Aborted

If[TrueQ[$Debug],
	handleDirtyException[e_] := Print["Bad arg to handleDirtyException: ", e]
]

(* Clean exceptions are those that are thrown before anything has been sent on the link. *)
handleCleanException[e_] :=
	(
		Assert[MatchQ[e, _manualException | _autoException]];
		reportException[e];
		$Failed
	)

(* Post exceptions are those that are thrown after the result has been completely sent. *)
handlePostException[e_] :=
	(
		Assert[MatchQ[e, Null | _manualException | _autoException]];
		If[e =!= Null,
			reportException[e, True];
		]
	)

(* Strategy for the exception-reporting functions is as follows: If user has defined a $JavaExceptionHandler,
   use it, otherwise use $internalJavaExceptionHandler, which has a default value (issue as a Message).
*)

$internalJavaExceptionHandler =
    If[ValueQ[$MessagePrePrint],
        Block[{oldMsgPrePrint = $MessagePrePrint, $MessagePrePrint},
            $MessagePrePrint = If[StringLength[#] > 2000, oldMsgPrePrint[#], #]&;
            Message[MessageName[#1, #2], ##3]
        ],
    (* else *)
        Message[MessageName[#1, #2], ##3]    
    ]&


reportException[e_manualException, isPost:(True | False):False] := reportException[First[e], isPost]
reportException[e_autoException, isPost:(True | False):False] := reportException[First[e], isPost]
reportException[e_String, isPost:(True | False)] :=
	Module[{excHandler, str = e},
		excHandler = If[ValueQ[$JavaExceptionHandler], $JavaExceptionHandler, $internalJavaExceptionHandler];
		(* Sometimes get newline or 0x00 chars at end of message. Remove them, and also the terminating
		   period if there is one.
		*)
		str = FromCharacterCode[ToCharacterCode[e] /. {a__, junk___?(#<=13&)} :> {a}];
		If[StringTake[str, -1] === ".", str = StringDrop[str, -1]];
		If[isPost, 
			excHandler[Java, "pexcptn", str],
		(* else *)
			excHandler[Java, "excptn", str]
		]
	]

End[]
