(* :Title: InstallJava.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 3.1 *)

(* :Mathematica Version: 4.0 *)
		     
(* :Copyright: J/Link source code (c) 1999-2005, Wolfram Research, Inc. All rights reserved.

   Use is governed by the terms of the J/Link license agreement, which can be found at
   www.wolfram.com/solutions/mathlink/jlink.
*)

(* :Discussion:
   InstallJava, UninstallJava and related.
	
   This file is a component of the J/Link Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   J/Link uses a special system wherein one package context (JLink`) has its implementation
   split among a number of .m files. Each component file has its own private context, and also
   potentially introduces public symbols (in the JLink` context) and so-called "package" symbols,
   where the term "package" comes from Java terminology, referring to symbols that are visible
   everywhere within the implementation of J/Link, but not to clients. The JLink.m file itself
   is produced by an automated tool from the component files and contains only declarations.
   
   Do not modify the special comment markers that delimit Public- and Package-level exports.
*)


(*<!--Public From InstallJava.m

InstallJava::usage =
"InstallJava[] launches the Java runtime and prepares it to be used from Mathematica. Only one Java runtime is ever launched; subsequent calls to InstallJava after the first have no effect."

StartJava::usage =
"StartJava is deprecated. Use InstallJava instead."

UninstallJava::usage =
"UninstallJava[] shuts down the Java runtime that was started by InstallJava. It is provided mainly for developers who are actively recompiling Java classes for use in Mathematica and therefore need to shut down and restart the Java runtime to reload the modified classes. Users generally have no reason to call UninstallJava. The Java runtime is a shared resource used by potentially many Mathematica programs. You should leave it running unless you are absolutely sure you need to shut it down."

QuitJava::usage =
"QuitJava is deprecated. Use UninstallJava instead."

ReinstallJava::usage =
"ReinstallJava[] is a convenience function that calls UninstallJava followed by InstallJava. It takes the same arguments as InstallJava. See the usage messages for InstallJava and UninstallJava for more information."

RestartJava::usage =
"RestartJava is deprecated. Use ReinstallJava instead."

JavaLink::usage =
"JavaLink[] returns the MathLink LinkObject that is used to communicate with the J/Link Java runtime. It will return Null if Java is not running."

JavaUILink::usage =
"JavaUILink[] returns the MathLink LinkObject used by calls to Mathematica that originate from Java user-interface actions, or Null if no such link is present."

ClassPath::usage =
"ClassPath is an option to InstallJava that controls whether the Java runtime should include the contents of the CLASSPATH environment variable in its class search path. The default is ClassPath->Automatic, which means to include CLASSPATH. If you specify ClassPath->None, CLASSPATH will be ignored. You can also specify a string giving a classpath specification in the standard platform-specific notation. Users considering specifying a string as the value for the ClassPath option should probably use the more flexible AddToClassPath function instead. The main use for the ClassPath option is to set it to None, in case you want to specifically prevent J/Link from including the contents of the CLASSPATH variable in its search path."

CommandLine::usage =
"CommandLine is an option to InstallJava that specifies the first part of the command line that should be used to launch the Java runtime. You can use this option to specify a name other than \"java\" for the Java runtime, or if you have more than one version of Java installed and you need to specify the full path to the runtime you want launched. The value of this option does not specify the entire command line, however, as InstallJava may add some arguments to the end of the line, depending on other options."

JVMArguments::usage =
"JVMArguments is an option to InstallJava that allows you to specify additional command-line arguments passed to the Java virtual machine at startup. The string you specify is added to the command line used to launch Java. You can use this option to specify properties with the standard -D syntax, such as \"-Dsome.property=true\". This option is not supported on Mac OSX."

-->*)

(*<!--Package From InstallJava.m

autoClassPath

$inPreemptiveCallFromJava

(* Experimental features for 6.0 *)
InstallJavaInternal
AsyncInstall

(* temporary *)
$InternalLink

-->*)


(* Current context will be JLink`. *)

Begin["`InstallJava`Private`"]


(* The options that deal with classpath are intended for users, not developers. In other words, you probably
    won't distribute code that uses these options. For one thing, you can't currently specify paths in a
    cross-platform way. Developers will have to tell their users to put their Java classes in a blessed place,
    or have them tweak their CLASSPATH variable.
*)

InstallJava::fail = "A link to the Java runtime could not be established."

InstallJava::uifail = "The separate Java user-interface link could not be established."

InstallJava::path =
"The J/Link package appears to be improperly installed. The package file is not in an appropriate location in relation to Mathematica's $Path."

InstallJava::opt = "Warning: unrecognized option in InstallJava."

Java::init = "Java is not running. You must call InstallJava[] to start the Java runtime."


(* StartJava/QuitJava/RestartJava are synonyms. *)
StartJava = InstallJava
QuitJava = UninstallJava
RestartJava = ReinstallJava

(* I generally don't protect options from redefinition if the package is read in twice, but the options
   to InstallJava are important, and users are encouraged to set them in their init.m if necessary. We
   don't want them to get redefined accidentally.
*)
If[Options[InstallJava] === {},
	Options[InstallJava] = {ClassPath->Automatic, CommandLine->Automatic, JVMArguments->None}
]

InstallJava[opts___?OptionQ] :=
	Module[{thisPkgPath, CP, cmdLine, jvmArgs, launcherLink, macLink, err},
		(* Bail out right away if link is already open and OK. *)
		If[MemberQ[Links[], $jlink],
			LinkReadyQ[$jlink]; (* Hit link to force LinkError to give current value. *)
			err = First[LinkError[$jlink]];
			(* Error 10 is "deferred connection still unconnected", and we allow it because we allow calls
			   into Java to hang until the link becomes connected by an asynchronous means.
			*)
			If[err === 0 || err === 10,
				Return[$jlink],
			(* else *)
				UninstallJava[]
			],
		(* else *)
			(* This extra test (jlink has a value, but it is not in Links[]) is to catch cases where
			   user has improperly shut down Java (e.g., by calling LinkClose).
			*)
			If[Head[$jlink] === LinkObject,
				(* Used to issue a warning message here saying that the previous session was improperly shut down.
				   Decided that was more irritating than useful, as it only shows up when you force-quit the old session.
				*)
				resetMathematica[]
			]
		];
		If[Length[{filterOptions[InstallJava, Flatten[{opts}]], filterOptions[LinkOpen, Flatten[{opts}]]}] != Length[Flatten[{opts}]],
			Message[InstallJava::opt]
		];
		{cmdLine, CP, jvmArgs} =
			{CommandLine, ClassPath, JVMArguments} /. Flatten[{opts}] /. Options[InstallJava];
		If[osIsMacOSX[],
			$jlink = Install[ToFileName[{$jlinkDir, "JLink.app", "Contents", "MacOS"}, "JavaApplicationStub"]],
		(* else *)
			(* Windows, UNIX *)
			If[$DebugCommandLine, Print["cmdline = ", createCommandLine[cmdLine, jvmArgs]]];
			(* On Windows and Unix, LinkQuote is not needed, and can interfere
			   with finding the program to launch (e.g., if CommandLine option has a full pathname with backslashes).
			   Because LinkQuote is wrapped around another function we don't want called (FindFile), we replace it
			   with Hold. 
			*)
			Block[{System`Dump`LinkQuote = Hold},
				$jlink = Install[createCommandLine[cmdLine, jvmArgs], filterOptions[LinkOpen, opts]]
			]
		];
		If[Head[$jlink] === LinkObject,
			MathLink`LinkAddInterruptMessageHandler[$jlink];
			$uilink = initJava[CP, useUILinkQ[]]
        ];
		If[Head[$jlink] === LinkObject && (Head[$uilink] === LinkObject || !useUILinkQ[]),
			$jlink,
		(* else *)
			Message[InstallJava::fail];
			If[Head[$jlink] === LinkObject, LinkClose[$jlink]];
			If[Head[$uilink] === LinkObject, LinkClose[$uilink]];
            $jlink = Null;
            $uilink = Null;
			$Failed
		]
	]

InstallJava[link_LinkObject] :=
	Module[{isParentLink, useUILink},
		(* Note that we don't call QuitJava here. It is up to the user of this advanced version to
		   make sure that QuitJava was called last time. This version is not safe to call whenever
		   you want.
		*)
		(* Don't want to set up uiLink if we are calling InstallJava[$ParentLink] from Java (this happens
		   during KernelLink.enableObjectReferences()). In that case there won't be a Reader thread, so
		   we don't want the uiLink.
		*)
		useUILink = useUILinkQ[] && link =!= $ParentLink;
		If[Head[$jlink] === LinkObject,
			resetMathematica[]
		];
		$jlink = Install[link];
		If[Head[$jlink] === LinkObject,
			MathLink`LinkAddInterruptMessageHandler[$jlink];
			$uilink = initJava[Automatic, useUILink]
        ];
		If[Head[$jlink] === LinkObject && (Head[$uilink] === LinkObject || !useUILink),
			$jlink,
		(* else *)
			Message[InstallJava::fail];
			If[Head[$jlink] === LinkObject, LinkClose[$jlink]];
			If[Head[$uilink] === LinkObject, LinkClose[$uilink]];
			$jlink = Null;
            $uilink = Null;
			$Failed
		]
	]

UninstallJava[] :=
	Module[{res = Null},
		If[Head[$jlink] === LinkObject,
			(* To avoid potentially many errors, only call onUnloadClass methods if jlink is alive and well. *)
			If[MemberQ[Links[], $jlink],
				LinkReadyQ[$jlink]; (* Hit link to force LinkError to give current value. *)
				If[First[LinkError[$jlink]] === 0,
					callAllUnloadClassMethods[]
				]
			];
			UnshareFrontEnd[JavaLink[]];
			If[MemberQ[SharingLinks[], JavaLink[]],
				UnshareKernel[JavaLink[]]
			];
			resetMathematica[];
			res = Uninstall[$jlink];
			$jlink = Null;
            If[Head[$uilink] === LinkObject,
				MathLink`RemoveSharingLink[$uilink];
                LinkClose[$uilink]
            ];
            $uilink = Null
		];
		res
	]


Options[ReinstallJava] = Options[InstallJava]

ReinstallJava[args___] := (UninstallJava[]; InstallJava[args])


If[!ValueQ[$jlink], $jlink = Null]
If[!ValueQ[$uilink], $uilink = Null]

JavaLink[] := $jlink

JavaUILink[] := $uilink


createCommandLine[cmdLine_, jvmArgs_] :=
	Module[{jlinkPath, cpSpec, prefsSpec, javaCmd, extraArgs},
		jlinkPath = 
			Scan[If[FileType[#] === File, Return[#]]&,
				{ToFileName[$jlinkDir, "JLink.jar"],  (* First try to find JLink.jar next to JLink.m *)
				 ToFileName[{$TopDirectory, "AddOns", "JLink"}, "JLink.jar"],  (* For >= 4.2 *)
				 ToFileName[{$TopDirectory, "AddOns", "Applications", "JLink"}, "JLink.jar"],
				 ToFileName[{$PreferencesDirectory, "Applications", "JLink"}, "JLink.jar"],  (* For >= 4.2 *)
				 ToFileName[{$PreferencesDirectory, "AddOns", "Applications", "JLink"}, "JLink.jar"]}
				(* OK for none to match--user may have put Jlink.jar into jre/lib/ext, or somewhere on CLASSPATH. *)
			];
		If[$VersionNumber >= 4.2,
			(* For 4.2 and later, a Java runtime is bundled with Mathematica on some platforms. *)
			javaCmd = 
				Which[
					StringQ[cmdLine],
						cmdLine,
					StringQ[Environment["WRI_JAVA_HOME"]] && FileType[Environment["WRI_JAVA_HOME"]] === Directory,
						ToFileName[{Environment["WRI_JAVA_HOME"], "bin"}, If[osIsWindows[], "javaw", "java"]],
					$SystemID == "Windows" || $SystemID == "Windows-x86-64",
						If[FileType[# <> ".exe"] === File, #, "javaw"]& @
							ToFileName[{$TopDirectory, "SystemFiles", "Java", $SystemID, "bin"}, "javaw"],
					True,
						(* Fallthrough for UNIX. *)
						If[FileType[#] === File, #, "java"]& @
							ToFileName[{$TopDirectory, "SystemFiles", "Java", $SystemID, "bin"}, "java"]
				],
		(* else *)
			(* V4.0 or 4.1 *)
			javaCmd = 
				Which[
					StringQ[cmdLine],
						cmdLine,
					$SystemID == "Windows",
						"javaw",
					True,
						(* UNIX, Linux *)
						"java"
				]
		];
		(* The command-line classpath spec points only at JLink.jar. All other class locations (including from CLASSPATH)
		   are specified later. If we did not find JLink.jar with our search above, however, we must not include any classpath spec
		   on the command line, as JLink.jar might need to be found off of CLASSPATH.
		*)
		cpSpec = If[!StringQ[jlinkPath], "", " -classpath \"" <> jlinkPath <> "\""];
		(* Disabling the Java prefs subsystem on Unix/Linux is a hack to work around a very annoying problem with
		   that subsystem. Maybe Sun will fix this in JDK 1.5. Because this is an experimental fix, we'll put in
		   the $disablePrefs flag as a backdoor that could be set from top level before launching Java.
		*)
		prefsSpec =
			If[!osIsWindows[] && !osIsMacOSX[] && $disablePrefs =!= False,
				" -Djava.util.prefs.PreferencesFactory=com.wolfram.jlink.DisabledPreferencesFactory",
			(* else *)
				""
			];
		extraArgs = If[StringQ[jvmArgs], " " <> jvmArgs, ""];
		javaCmd <> cpSpec <> extraArgs <> prefsSpec <> " com.wolfram.jlink.Install"
	]


initJava[CP_, setupUILink:(True | False)] :=
    Module[{link, prot, linkName},
    	jSetUserDir[$HomeDirectory];
		jAddToClassPath[autoClassPath[], True];
		(* Add CLASSPATH variable _after_ auto class path. We must add CLASSPATH manually since we are using the
		   -classpath command-line option to point solely at JLink.jar.
		*)
		If[CP === Automatic && StringQ[Environment["CLASSPATH"]],
			jAddToClassPath[splitClasspath[Environment["CLASSPATH"]], False]
		];
		(* Here we add the contents of the user-specified ClassPath option. *)
		If[StringQ[CP],
			jAddToClassPath[splitClasspath[CP], False]
		];
		jAddToClassPath[$ExtraClassPath, True];
		(* Now we add our bundled tools.jar. We put it at the end on the off chance that a user might want to
		   have another source for these classes loaded earlier. Tools.jar contains that Java compiler, among other things.
		*)
		jAddToClassPath[{ToFileName[{$TopDirectory, "SystemFiles", "Java", $SystemID, "lib"}, "tools.jar"]}, False];
        If[isPreemptiveKernel[] && setupUILink,
	        (* Set up the UI link. $UILinkProtocol exists as a backdoor for users who need to
	           force a particular protocol (e.g., TCP to avoid problems with TCPIP).
			*)
	        prot =
				Which[
	        		StringQ[$UILinkProtocol], $UILinkProtocol,
	        		osIsWindows[], "SharedMemory",
					True, "TCPIP"
				];
	        link = LinkCreate[LinkProtocol->prot];
	        (* On OS/X can have problems if name includes "@localhost", so remove it. See bug 58268. *)
	        linkName = StringReplace[First[link], "@localhost" -> ""];
	        If[TrueQ[jUILink[linkName, prot, TrueQ[$UseLinkSnooper]]],
	            LinkConnect[link];
				MathLink`AddSharingLink[link,
						MathLink`LinkSwitchPre -> linkSwitchPreFunc,
						MathLink`LinkSwitchPost -> linkSwitchPostFunc,
						MathLink`AllowPreemptive -> True,
						MathLink`ImmediateStart -> True
				];
				MathLink`LinkAddInterruptMessageHandler[link];
	            link,
	        (* else *)
	            Message[InstallJava::uifail];
	            LinkClose[link];
	            Null
	        ],
	    (* else *)
	    	(* Version 5.0 or earlier--no UI Link *)
	    	Null
	    ]
	]


If[!ValueQ[$inPreemptiveCallFromJava], $inPreemptiveCallFromJava = False]

(* In M 6.0 and later, the 2nd arg to this func tells whether this call is preemptive or not,
   but for 5.x compatibility I won't use it, and instead call MathLink`IsPreemptive[] instead.
*)
linkSwitchPreFunc[___] :=
	Block[{uiLink, oldFrontEnd, oldFormatType, oldEndDlgPktLink, res},
		uiLink = JavaUILink[];
		If[MathLink`IsPreemptive[], $inPreemptiveCallFromJava = True];
		oldFormatType = FormatType /. Options["stdout"];
		SetOptions["stdout", FormatType->OutputForm];
		oldFrontEnd = 
			If[hasServiceFrontEnd[],
				MathLink`SetServiceFrontEnd[],
			(* else *)
				If[FrontEndSharedQ[JavaLink[]],
					(* Note that for legacy reasons, users call ShareFrontEnd[JavaLink[]], but really
					   it is the JavaUILink[] that the FE-specific traffic goes out on.
					*)
					MathLink`SetFrontEnd[uiLink],
				(* else *)
					res = MathLink`SetFrontEnd[Null];
					MathLink`SetMessageLink[uiLink];
					res
				]
			];
		If[First[oldFrontEnd] === False,
			(* There was no ServiceLink, and the MessageLink was set to Null.
			   FE services won't work, but at least we can set the MessageLink
			   to the activeJavaLink, so that side-effect output will come to
			   Java and not get completely lost.
			*)
			MathLink`SetMessageLink[uiLink]
		];
		oldEndDlgPktLink = MathLink`$EndDialogPacketLink;
		MathLink`$EndDialogPacketLink = Null;
		{oldFrontEnd, oldFormatType, oldEndDlgPktLink}
	]
	
linkSwitchPostFunc[{oldFrontEnd_, oldFormatType_, oldEndDlgPktLink_}] :=
	(
		MathLink`RestoreFrontEnd[oldFrontEnd];
		SetOptions["stdout", FormatType->oldFormatType];
		MathLink`$EndDialogPacketLink = oldEndDlgPktLink;
		$inPreemptiveCallFromJava = False;
	)
	

(* Determines the default automatic set of extra directories to search for classes. Looks for Java subdirectories of
   any of several standard application directories.
*)
autoClassPath[] :=
	Module[{appPaths, appDirs, javaDirs},
		appPaths = {ToFileName[{$TopDirectory, "AddOns", "Applications"}],
					ToFileName[{$TopDirectory, "AddOns", "ExtraPackages"}],
					ToFileName[{$TopDirectory, "AddOns", "AutoLoad"}]};
		If[StringQ[$AddOnsDirectory],
			(* This branch is for 4.2 and later (4.1.5 on Mac OSX). *)
			appPaths =
				{ToFileName[{$UserAddOnsDirectory, "Applications"}],
				 ToFileName[{$UserAddOnsDirectory, "AutoLoad"}],
				 ToFileName[{$AddOnsDirectory, "Applications"}],
				 ToFileName[{$AddOnsDirectory, "AutoLoad"}]} ~Join~ appPaths,
		(* else *)
			PrependTo[appPaths, ToFileName[{$PreferencesDirectory, "AddOns", "Applications"}]]
		];
		appDirs = Select[Flatten[FileNames["*", #]& /@ appPaths], (FileType[#] === Directory)&];
		(* Now append $AddOns and $UserAddOns, to allow Java subdirs to be found (i.e., allow
		   $UserAddOnsDirectory/Java, not just $UserAddOnsDirectory/Applications/SomeApp/Java).
		   Append instead of prepend, to give applications primacy.
		*)
		If[StringQ[$AddOnsDirectory],
			appDirs = appDirs ~Join~ {$AddOnsDirectory, $UserAddOnsDirectory}
		];
		javaDirs = Select[ToFileName[{#, "Java"}]& /@ appDirs, (FileType[#] === Directory)&];
		(* Here we put the special WRI SystemFiles/Java dir first. The thinking for putting it absolutely first
		   in the search path is that it becomes a convenient place to put classes and be sure they override any
		   provided by applications. It can be used to resolve application conflicts in this way.
		*)
		PrependTo[javaDirs, ToFileName[{$TopDirectory, "SystemFiles", "Java"}]];
		javaDirs
	]


(* The Mathematica-side things that must be done when starting a fresh Java session. After this func is run, it should
    be true that the kernel is in the same state it was in before any Java sessions were started. The possible exception
    to this is that certain contexts may exist that weren't present before, but their contents should be empty. This function
    is only used during InstallJava; it is not user-visible, and no attempt is made to provide users with a way to
    "reset" their Mathematica-Java session.
*)
resetMathematica[] :=
	(
		clearJavaDefs[];
		resetJavaBlock[];
		clearComplexClass[];
		unregisterAllWindows["Java"];
	)


(* Splits a platform-specific classpath specification into a list of strings, one for each component. *)
splitClasspath[cp_String] :=
	Module[{cpSep, str, result},
		cpSep = If[osIsWindows[], ";", ":"];
		strm = StringToStream[cp]; 
		result = ReadList[strm, Word, WordSeparators->cpSep, NullWords->False];
		Close[strm];
		result
	]


(* Tells whether to set up the separate UI link. Don't want to setup uiLink if not a preemptive kernel, or a standalone kernel. *)
useUILinkQ[] :=
	isPreemptiveKernel[] && ValueQ[$ParentLink] && $ParentLink =!= Null;


(****************************)

AsyncInstall[] :=
	With[{loop = LinkOpen[LinkMode->Loopback]},
		(* Needs["JLink`"]; *)
		LinkWriteHeld[loop, Hold[$InternalLink = LinkCreate[]]];
		LinkWriteHeld[loop, Hold[InstallToInternalLink[$InternalLink]]];
		LinkWriteHeld[loop, Hold[InstallJava[$InternalLink]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["java.lang.Object"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["java.awt.Component"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["com.wolfram.guikit.GUIKitDriver"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["com.wolfram.bsf.engines.MathematicaBSFEngine"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["org.apache.bsf.util.BSFEngineImpl"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["com.wolfram.bsf.util.MathematicaBSFFunctions"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["java.io.FileInputStream"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["java.util.jar.Manifest"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["java.util.jar.Attributes"]]];
		LinkWriteHeld[loop, Hold[LoadJavaClass["com.wolfram.guikit.swing.GUIKitJFrame"]]];
		(* TODO: Move setlookandfeel in here. *)
		LinkWriteHeld[loop, Hold[LinkClose[loop]]];
		MathLink`AddSharingLink[loop];
	]

InstallJavaInternalDEVICE[] :=
	Module[{},
		$InternalLink = LinkCreate[];
		Java`InstallToInternalLink[$InternalLink];
		InstallJava[$InternalLink]
	]

InstallJavaInternalLOOPBACK[] :=
	Module[{res},
		$InternalLink = LinkOpen[LinkMode->Loopback];
		res = Java`InstallToInternalLink[$InternalLink];
		If[res === $Failed,
			Message[InstallJava::fail];
			Return[$Failed]
		];
		Block[{jlinkDefineExternal = jlinkDefineInternal},
			Install[$InternalLink]
		];
		JLink`InstallJava`Private`$jlink = $InternalLink
	]

(* This one gets used--swap to eiher LOOPBACK or DEVICE version. *)
InstallJavaInternal[] :=
	Module[{loop, res},
		loop = LinkOpen[LinkMode->Loopback];
		res = Java`InstallToInternalLink[loop];
		If[res === $Failed,
			Message[InstallJava::fail];
			Return[$Failed]
		];
		(**********
		Block[{jlinkDefineExternal = jlinkDefineInternal},
			Install[$InternalLink]
		];
		JLink`InstallJava`Private`$jlink = $InternalLink
		***********)
		$InternalLink = loop;
		If[True,  (* Full treatment *)
			InstallJava[loop],
		(* else *)  (* leaves out uilink, etc., for debugging. *)
			Install[loop];
			JLink`InstallJava`Private`$jlink = $InternalLink
		]
	]

jlinkDefineInternal[p_String, a_, n_] := 
	Module[{e, pat = ToHeldExpression[p], args = ToHeldExpression[a]}, 
		e = Hold[_ := jlinkInternalCall[$InternalLink, CallPacket[_, _]]];
		e = ReplaceHeldPart[e, pat, {1, 1}];
		e = ReplacePart[e, n, {1, 2, 2, 1}];
		e = ReplaceHeldPart[e, args, {1, 2, 2, 2}];
		ReleaseHold[e];
	]


(* TODO: Not sure about usaing AbortProtect here as in jlinkDefineExternal. That's a big issue... *)
jlinkInternalCall[link_LinkObject, packet_CallPacket] :=
	Block[{ThisLink = link, $CurrentLink = link, pkt = packet, res},
		While[True,
			If[LinkWrite[link, pkt] === $Failed, Return[$Failed]];
			Java`DispatchToJava[link];
			res = LinkReadHeld[link];
			Switch[res,
				Hold[EvaluatePacket[_]],
					(* Re-enable aborts during the computation in Mathematica of EvaluatePacket contents, but have
					   them just cause $Aborted to be returned to Java, not call Abort[].
					*)
					pkt = ReturnPacket[CheckAbort[res[[1,1]], $Aborted]],
				Hold[ReturnPacket[_]],
					Return[res[[1,1]]],
				Hold[_],
					Return[res[[1]]],
				_,
					Return[res]
			]
		]		
	]

(********************************)
(* Include private defs here to avoid reliance on the Utilities`FilterOptions` standard package (which is
   not included in a Minimal Install.
*)
filterOptions[command_Symbol, options___] := filterOptions[First /@ Options[command], options]
filterOptions[opts_List, options___] := Sequence @@ Select[Flatten[{options}], MemberQ[opts, First[#]]&]


End[]
