(* :Title: Misc.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 3.1 *)

(* :Mathematica Version: 4.0 *)
		     
(* :Copyright: J/Link source code (c) 1999-2005, Wolfram Research, Inc. All rights reserved.

   Use is governed by the terms of the J/Link license agreement, which can be found at
   www.wolfram.com/solutions/mathlink/jlink.
*)

(* :Discussion:
   Miscellaneous functions.
	
   This file is a component of the J/Link Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   J/Link uses a special system wherein one package context (JLink`) has its implementation
   split among a number of .m files. Each component file has its own private context, and also
   potentially introduces public symbols (in the JLink` context) and so-called "package" symbols,
   where the term "package" comes from Java terminology, referring to symbols that are visible
   everywhere within the implementation of J/Link, but not to clients. The JLink.m file itself
   is produced by an automated tool from the component files and contains only declarations.
   
   Do not modify the special comment markers that delimit Public- and Package-level exports.
*)



(*<!--Public From Misc.m

JavaShow::usage =
"JavaShow[windowObj] causes the specified Java window to be brought to the foreground, so that it appears in front of notebook windows. Use this function in place of standard Java window methods like setVisible (or its deprecated equivalent, show) and toFront. The effects those methods have are unfortunately not identical on different Java virtual machines and operating systems. JavaShow performs the required steps in VM-specific ways, so programs that use it will work identically on different configurations. The argument must be an object of a class that can represent a top-level window (i.e., an instance of the java.awt.Window class or a subclass)."

DoModal::usage =
"DoModal[] does not return until the Java side sends an expression of the form EvaluatePacket[EndModal[___]]. Use DoModal to have a program wait until some Java user-interface action signals that it can proceed. Typically, this would be used to implement a modal dialog box that needs to return a result to Mathematica when it is dismissed. DoModal returns the EndModal[args] expression."

EndModal::usage =
"EndModal is the head of an expression sent by Java to signal the end of a DoModal[] loop."

ServiceJava::usage =
"ServiceJava[] allows a single computation originating from Java to proceed. ServiceJava joins DoModal and ShareKernel as the means to allow computations to proceed that originate from Java. \"Originate\" from Java means that the call to Mathematica is in response to a user action in Java (like pressing a button), as opposed to a callback into Mathematica from a call to Java that originated from Mathematica. Use ServiceJava when you want a running Mathematica program to periodically allow computations from the Java user interface. ServiceJava, like ShareKernel, is unnecessary in Mathematica 5.1 and later because the kernel is always ready for computations that originate in Java, even when busy with another computation."

AppletViewer::usage =
"AppletViewer[class, params] displays a window with an applet of the named class running in it. The class can be given as a string or a JavaClass expression. The params argument is a list of strings, each specifying a name=value pair supplying applet properties that would normally appear in an HTML file in <PARAM> tags or other applet attributes (like WIDTH and HEIGHT attributes). A typical specification might be {\"WIDTH=400\", \"HEIGHT=400\"}."

PeekClasses::usage =
"PeekClasses[] returns a list of the classes currently loaded into Java by Mathematica. It is intended to be used only as a debugging aid."

PeekObjects::usage =
"PeekObjects[] returns a list of the Java objects that have been sent to Mathematica (and not yet released with ReleaseJavaObject). It is intended to be used only as a debugging aid."

SetInternetProxy::usage =
"SetInternetProxy[\"host\", port] sets proxy information in your Java session for accessing the Internet. If you use a proxy for accessing the Internet, you may need to call this function to enable Java code to use the Internet. Consult your network administrator for proxy settings. A typical example would look like SetInternetProxy[\"proxy.mycompany.com\", 8080]."

ShowJavaConsole::usage =
"ShowJavaConsole[] displays the Java console window and begins capturing output sent to the Java System.out and System.err streams. Anything written to these streams before ShowJavaConsole is first called will not appear, but subsequent output will be captured even if the window is closed and reopened. Only the most recent one thousand lines of output are displayed. ShowJavaConsole[\"stdout\"] captures only System.out, and ShowJavaConsole[\"stderr\"] captures only System.err."

JavaGC::usage =
"JavaGC[] requests the Java garbage collector to be run. It returns the number of bytes freed. JavaGC invokes the Java method Runtime.getRuntime().gc()."

InstanceOf::usage =
"InstanceOf[javaobject, javaclass] gives True if javaobject is an instance of the class or interface javaclass, or a subclass. Otherwise, it returns False. It mimics the behavior of the Java language's instanceof operator. The javaclass argument can be either the fully-qualified class name as a string, or a JavaClass expression."

SameObjectQ::usage =
"SameObjectQ[javaobject1, javaobject2] returns True if and only if the JavaObject expressions javaobject1 and javaobject2 refer to the same Java object. In other words, it behaves like the Java language's == operator as applied to object references."

JavaThrow::usage =
"JavaThrow[\"exceptionClassName\", \"detailMsg\"] causes the specified exception to be thrown in the Java thread that called the Mathematica program in which JavaThrow occurred. You can also call JavaThrow with an Exception object created by a call to JavaNew, instead of a string giving the name of the exception class. If you use the string form, the second argument is an optional detail message for the exception. It is up to the user to ensure that the exception being created has an appropriate constructor. JavaThrow is a specialized function that few programmers will have any use for."

GetClassPath::usage =
"GetClassPath[] is deprecated. Use JavaClassPath[] instead."

JavaClassPath::usage =
"JavaClassPath[] returns the class search path in use by the Java runtime. This includes classes specified via the CLASSPATH environment variable (if any), directories and files added by the user with AddToClassPath, and those directories automatically searched by J/Link. The result is a list of strings specifying directories and .jar or .zip files, in the same order that they will be searched for classes by the Java runtime."

AddToClassPath::usage =
"AddToClassPath[\"location1\", \"location2\", ...] adds the specified full paths to directories and jar or zip files to the J/Link class search path. If you specify a directory, it will be searched automatically for .jar and .zip files, in addition to appropriately nested .class files, so you do not have to name .jar or .zip files explicitly. Changes take effect immediately and apply only to the currently-running Java runtime. If you quit and restart Java, you will need to call AddToClassPath again."

$ExtraClassPath::usage =
"$ExtraClassPath is deprecated. Use the function AddToClassPath instead."

SetComplexClass::usage =
"SetComplexClass[\"classname\"] specifies the Java class to use for complex numbers sent from, and returned to, Mathematica."

GetComplexClass::usage =
"GetComplexClass[] returns the Java class used for complex numbers sent from, and returned to, Mathematica."

ParentClass::usage =
"ParentClass[javaclass] returns the JavaClass expression representing the parent class of javaclass. You can also specify an object of a class, as in ParentClass[javaobject]."

ClassName::usage =
"ClassName[javaclass] returns, as a string, the fully-qualified name of the Java class represented by javaclass. You can also specify an object of the class, as in ClassName[javaobject]."

GetClass::usage =
"GetClass[javaobject] returns the JavaClass that identifies the object's class."

ImplementJavaInterface::usage =
"ImplementJavaInterface[interfaces, handlerMappings] uses the Dynamic Proxy facility of Java to create a new Java class and return an object of that class that implements the named interface or list of interfaces by calling back into Mathematica. In short, it lets you create a Java object that implements a given Java interface entirely in Mathematica code. The handlerMappings argument is a list of rules that specify the name of the Java method and the name of the method that will be called in Mathematica to implement the body of that method, as in \"intfMeth1\"->\"mathHandlerMethod1\". The arguments passed to the Mathematica method will be exactly the arguments originally passed to the Java method."

JavaWindowToFront::usage =
"JavaWindowToFront is an internal symbol."

-->*)

(*<!--Package From Misc.m

clearComplexClass
osIsWindows
osIsClassicMac
osIsMacOSX

isPreemptiveKernel
hasFrontEnd
hasServiceFrontEnd
hasLocalFrontEnd

/// Exposed for the benefit of .NET/Link. Not used outside this file by J/Link.
registerWindow
unregisterWindow
windowID
unregisterAllWindows
windowRegisteredQ
getNextWindowID

/// Called from Java
titleChangedFunc

/// Experimental utility func.
asyncEvaluate

contextIndependentOptions

-->*)


(* Current context will be JLink`. *)

Begin["`Misc`Private`"]


(****************************************  Funcs that call Java directly  ******************************************)

SameObjectQ::obj = "At least one argument to SameObjectQ was not a valid reference to a Java object or Null."

(* Wrap in TrueQ to ensure T/F result even on exception. *)
SameObjectQ[obj1_?JavaObjectQ, obj2_?JavaObjectQ] := TrueQ[jSameQ[obj1, obj2]]

SameObjectQ[_, _] := (Message[SameObjectQ::obj]; False)


JavaGC[] := jGC[]


SetComplexClass[clsName_String] := 
	Module[{cls = LoadJavaClass[clsName, Null, True]},
		If[Head[cls] === JavaClass,
			SetComplexClass[cls],
		(* else *)
			$Failed
		]
	]

(* It is too expensive to call into Java to get the complex class, so I have to store it in
   Mathematica as well. This means that user must not let the values stored in Java and Mathematica
   get out of sync. The only way to do that would be to manually call setComplexClass() in Java.
*)

SetComplexClass[jc_JavaClass] :=
	If[jSetComplex[classIndexFromClass[jc]] =!= Null,
		$complexClass =.;
		$Failed,
	(* else *)
		$complexClass = jc;
	]

GetComplexClass[] := If[ValueQ[$complexClass], $complexClass, Null]

clearComplexClass[] := Clear[$complexClass]


GetClassPath = JavaClassPath  (* GetClassPath is deprecated. *)

JavaClassPath[] :=
	Module[{res = jClassPath[]},
		If[osIsWindows[],
			(* On Windows, files and dirs come back looking like /C:/path/to/dir/. Fix these up to avoid confusing users, and
			   to put entries into a format Mathematica functions can understand. 
			*)
			res = If[StringMatchQ[#, "/*:*"], StringDrop[StringReplace[#, "/" -> "\\"], 1], #]& /@ res
		];
		res
	]

AddToClassPath[locs__String] := AddToClassPath[{locs}]
AddToClassPath[locs:{__String}] :=
	(
		InstallJava[];
		jAddToClassPath[locs, True];
		JavaClassPath[]
	)

(* $ExtraClassPath is deprecated, but still works. *)
If[!MatchQ[$ExtraClassPath, {___String}],
	$ExtraClassPath = {}
]


(* Call with either a Java exception object or the string name of the exception class you want to throw. In the first case, supply
   no second arg (it will be ignored); in the second case, the second arg is an optional detail message for the exception.
   It is up to the user to ensure that the exception being created has an appropriate zero- or one-arg constructor.
   This function is still experimental.
*)
JavaThrow[exc_, msg_String:""] := jThrow[exc, msg]


PeekClasses[] := classFromIndex /@ jPeekClasses[]
PeekObjects[] := jPeekObjects[]


releaseAllObjects[classIndex_Integer] := jReleaseAllObjects[classIndex]


(*********************************************  Public utility funcs  ***********************************************)

SetAttributes[ClassName, Listable]

ClassName[Null] = Null   (* Should this issue a message? *)

ClassName[cls_JavaClass] := classNameFromClass[cls]

ClassName[obj_?JavaObjectQ] := classNameFromIndex[classIndexFromInstance[obj]]


SetAttributes[ParentClass, Listable]

ParentClass[Null] = Null   (* Should this issue a message? *)

ParentClass[obj_?JavaObjectQ] := ParentClass[GetClass[obj]]

ParentClass[cls_String] := ParentClass[LoadJavaClass[cls]]

ParentClass[cls_JavaClass] :=
	If[# === Null,
		(* cls was java.lang.Object *)
		Null,
	(* else *)
		classFromIndex[#]
	]& [parentClassIndexFromClassIndex[classIndexFromClass[cls]]]


SetAttributes[GetClass, Listable]

GetClass[Null] = Null   (* Should this issue a message? *)

GetClass[obj_?JavaObjectQ] := classFromIndex[classIndexFromInstance[obj]]


InstanceOf::obj = "`1` is not a valid Java or .NET object reference."
InstanceOf::cls = "Invalid class specification `1`."

InstanceOf[Null, _String | _JavaClass] = False
InstanceOf[obj_?JavaObjectQ, cls_JavaClass] := InstanceOf[obj, classNameFromClass[cls]]
InstanceOf[obj_?JavaObjectQ, cls_String] := TrueQ[jInstanceOf[obj, cls]]
InstanceOf[obj_, cls_] := 
    Module[{isJava, isNET},
        isJava = JavaObjectQ[obj];
        isNET = NETLink`NETObjectQ[obj];
        If[!isJava && !isNET,
            Message[InstanceOf::obj, obj],
        (* else *)
            Message[InstanceOf::cls, cls]
        ];
        False
    ]


(********************************************  ImplementJavaInterface  **********************************************)

ImplementJavaInterface::jdk13 =
"The ImplementJavaInterface function requires JDK 1.3 or later. Your Java version does not appear to be modern enough."

ImplementJavaInterface[intfs:{__String}, handlerMappings__Rule] := ImplementJavaInterface[intfs, {handlerMappings}]
ImplementJavaInterface[intf_String, handlerMappings__Rule] := ImplementJavaInterface[{intf},{handlerMappings}]
ImplementJavaInterface[intf_String, handlerMappings:{__Rule}] := ImplementJavaInterface[{intf}, handlerMappings]

ImplementJavaInterface[intfs:{__String}, handlerMappings:{__Rule}] :=
	Module[{invHandler, interfaces, proxy},
		If[LoadJavaClass["java.lang.reflect.Proxy"] === $Failed,
			Message["ImplementJavaInterface::jdk13"];
			Return[$Failed]
		];
		LoadJavaClass["com.wolfram.jlink.JLinkClassLoader"];
		JavaBlock[
			interfaces = Symbol["com`wolfram`jlink`JLinkClassLoader`classFromName"] /@ intfs;
			If[!(And @@ (JavaObjectQ /@ interfaces)),
				(* Error messages will already have been issued by classFromName(). *)
				Return[$Failed]
			];
			invHandler = JavaNew["com.wolfram.jlink.MathInvocationHandler", handlerMappings /. Rule->List];
			Proxy`newProxyInstance[interfaces[[1]]@getClassLoader[], interfaces, invHandler]
		]
	]


(*************************************  SetInternetProxy  ************************************************)

SetInternetProxy[host_String, port_Integer] :=
	JavaBlock[
		InstallJava[];
		LoadJavaClass["java.lang.System"];
		System`getProperties[]@put[MakeJavaObject["proxySet"], MakeJavaObject["true"]];
		System`getProperties[]@put[MakeJavaObject["proxyHost"], MakeJavaObject[host]];
		System`getProperties[]@put[MakeJavaObject["proxyPort"], MakeJavaObject[ToString[port]]];
	]


(********************************************  AppletViewer  ************************************************)

AppletViewer[cls:(_String | _JavaClass)] := AppletViewer[cls, {}]

AppletViewer[cls:(_String | _JavaClass), params:{___String}] :=
	JavaBlock[
		AppletViewer[JavaNew[cls], params]
	]

AppletViewer[applet_?JavaObjectQ] := AppletViewer[applet, {}]

AppletViewer[applet_?JavaObjectQ, params:{___String}] :=
	JavaBlock[
		JavaNew["com.wolfram.jlink.MathAppletFrame", applet, params] // JavaShow;
	]


(************************************************  DoModal  *****************************************************)

(* DoModal enters a loop that waits until the Java side sends back
		EvaluatePacket[EndModal[...args for event listener...]]
   The loop it runs is capable of servicing normal EvaluatePacket requests. It is really a modified
   version of the ExternalAnswer function that handles reads during a normal call into Java.
   
   Note that DoModal turns off preemptive functions in a 5.1+ kernel. The links that use DoModal
   are "service links", meaning that they are used for calls that originate in M at unexpected times.
   We always need to turn off preemption when using a service link because a preemptive call could try
   to reentrantly use a service link.
*)

DoModal[] := DoModal[getActiveJavaLink[]]

DoModal[link_LinkObject] := doModal[link]

(* EndModal should not be called with arguments, as they will be ignored, but I want to make sure that it
   ends the loop even if users call it incorrectly.
*)
EndModal[___] := (endModal = True;)

doModal[link_LinkObject] :=
	AbortProtect[
		Block[{result = Null, endModal = False, e},
			If[link === JavaLink[] || link === JavaUILink[], jAllowUIComputations[True, True]];
	    	While[!TrueQ[endModal],
				e = LinkReadHeld[link];
				Assert[MatchQ[e, Hold[EvaluatePacket[_]]]];
				(* This 2+2 eval is a hack to absorb any aborts that might be pending here,
				   so they do not interfere with a call to EndModal[] that might occur during
				   the evaluation of e.
				*)
				CheckAbort[2+2, $Aborted];
				result = CheckAbort[e[[1,1]], $Aborted];
				If[LinkWrite[link, ReturnPacket[result]] === $Failed, 
					Return[$Failed]
				]
			];
			If[link === JavaLink[], jAllowUIComputations[False, False]];
			result
		]
	]

(* This allows a single computation originating from the Java UI thread (or any other non-Reader thread)
   to proceed.
*)
ServiceJava[] :=
	If[isPreemptiveKernel[],
		(* A no-op for preemptive kernel, because comps come in on the UI link. Because
		   we have taken out a read/write to Java (in myLinkReadyQ), we have lost some yielding
		   to Java, and programs that use ServiceJava probably keep the kernel busy to the detriment
		   of the Java UI thread. Therefore we Pause here for a tiny bit in an attempt to make
		   legacy programs written with ServiceJava behave as responsively as they used to.
		   Programs targeted at 5.1 and later have no need to call ServiceJava, and they can Pause
		   manually if desired to maintain smoothness in Java, say for an animation.
		*)
		Pause[.01],
	(* else *)
		AbortProtect[
			Module[{e, result, jl = JavaLink[]},
				If[myLinkReadyQ[jl],
					jAllowUIComputations[True, False]; 
					e = LinkReadHeld[jl];
					Assert[MatchQ[e, Hold[EvaluatePacket[_]]]];
					result = CheckAbort[e[[1,1]], $Aborted];
					LinkWrite[jl, ReturnPacket[result]]
				]
			]
		]
	]
	

(*******************************************  JavaShow  ***********************************************)

JavaShow::wnd = "The object passed to JavaShow must be a subclass of java.awt.Window."

JavaShow[windowObj_?JavaObjectQ] :=
	JavaBlock[
		Module[{id, title},
			If[!InstanceOf[windowObj, "java.awt.Window"],
				Message[JavaShow::wnd];
				Return[$Failed]
			];
			jShow[windowObj];
			If[!windowRegisteredQ[windowObj] && hasLocalFrontEnd[] && hasServiceFrontEnd[],
				id = If[osIsWindows[], jGetWindowID[windowObj], getNextWindowID[]];
				If[!TrueQ[id > 0],
					(* Failure in JAWT code to get HWND. Fail silently. *)
					Return[]
				];
				If[InstanceOf[windowObj, "java.awt.Frame"],
					title = windowObj@getTitle[]
				];
				If[SymbolQ[title] || title == "", title = "Java Window"];
				registerWindow[windowObj, title, "Java", id];
				windowObj@addWindowListener[JavaNew["com.wolfram.jlink.MathWindowListener", {{"windowClosing", ToString[windowClosingFunc]}}]];
				jAddTitleChangeListener[windowObj, ToString[titleChangedFunc]];
				KeepJavaObject[windowObj, Manual]
			];
		]
	]


windowClosingFunc[evt_] :=
	JavaBlock[
		unregisterWindow[evt@getWindow[]];
		ReleaseJavaObject[evt];
	]

titleChangedFunc[windowObj_, newTitle_] :=
	If[windowRegisteredQ[windowObj],
		MathLink`CallFrontEnd[FrontEnd`AttachWindow[newTitle, "Java", ToString[windowID[windowObj]]]];
	]


(* Called by FE when user selects Java window from Window menu. To keep the FE responsive,
   this is written so as not to block. Otherwise, it would hang the FE if the user selected
   a Java window from the FE Window menu while a long Java computation was going on
   (calls on JavaLink[] cannot preempt each other, so the call to jShow[] would block
   until the previous Java computation was finished).
*)
JavaWindowToFront[id_String] :=
	asyncEvaluate[
		Module[{windowList},
			windowList = Cases[registeredWindows[], {obj_, ToExpression[id], "Java"} -> obj];
			If[Length[windowList] > 0,
				JavaShow[First[windowList]]
			]
		]
	]
	

(* The following funcs deal with registering windows with the FE so they
   appear in its Windows menu. These funcs are used by .NET/Link also.
*)

(* Returns False if window is already registered. *)
registerWindow[obj_, title_String, type_String, id_Integer] :=
	If[!windowRegisteredQ[obj],
		MathLink`CallFrontEnd[FrontEnd`AttachWindow[title, type, ToString[id]]];
		AppendTo[$registeredWindows, {obj, id, type}];
		True,
	(* else *)
		(* Already registered. *)
		False
	]
	
unregisterWindow[obj_] :=
	Module[{windowRec, id},
		windowRec = Cases[$registeredWindows, {obj, _, _}];
		If[Length[windowRec] > 0,
			(* Should always have length 1. *)
			windowRec = First[windowRec];
			id = windowRec[[2]];
			If[hasServiceFrontEnd[],
				MathLink`CallFrontEnd[FrontEnd`DetachWindow[ToString[id]]]
			];
			$registeredWindows = DeleteCases[$registeredWindows, {obj, id, _}]
		]
	]

(* Type will be "Java" or ".NET". This func is called in UninstallJava/NET. *)
unregisterAllWindows[type_String] :=
	unregisterWindow /@ First /@ Cases[registeredWindows[], {_, _, type}]

windowRegisteredQ[obj_] := MemberQ[First /@ registeredWindows[], obj]

windowID[obj_] := If[windowRegisteredQ[obj], First[Cases[registeredWindows[], {obj, id_, _} -> id]], -1]

getNextWindowID[] := $nextWindowID++

registeredWindows[] := $registeredWindows

If[!ValueQ[$nextWindowID], $nextWindowID = 1]
If[!ValueQ[$registeredWindows], $registeredWindows = {}]


(**********************************  asyncEvaluate  **************************************)

(* Evaluates an expr at a later time. The expr is evaluated non-preemptively and treated like input
   arriving on a non-preemptive sharing link.
*) 

SetAttributes[asyncEvaluate, HoldFirst]

asyncEvaluate[e_] :=
	With[{loop = LinkOpen[LinkMode->Loopback]},
		LinkWriteHeld[loop, Hold[e]];
		LinkWriteHeld[loop, Hold[LinkClose[loop]]];
		MathLink`AddSharingLink[loop];
	]


(*******************************************  ShowJavaConsole  ***********************************************)

ShowJavaConsole[] := ShowJavaConsole["stdout", "stderr"]

ShowJavaConsole[None] := ShowJavaConsole["none"]

ShowJavaConsole[strms__String] :=
	JavaBlock[
		Module[{console},
			InstallJava[];
			console = jGetConsole[];
			If[console@isFirstTime[],
				console@setSize[450, 400];
				console@setLocation[100, 100];
				console@setFirstTime[False]
			];
			(* Plus should really be BitOr, but I don't want to introduce an unnecessary M-- 3.x incompatibility. *)
			console@setCapture[Plus @@ (Union[{strms}] /.
				{"stdout" -> Symbol["com`wolfram`jlink`ui`ConsoleWindow`STDOUT"],
				 "stderr" -> Symbol["com`wolfram`jlink`ui`ConsoleWindow`STDERR"],
				 "none" -> Symbol["com`wolfram`jlink`ui`ConsoleWindow`NONE"]})
			];
			JavaShow[console];
			console
		]
	]


(*****************************************  Platform tests  *******************************************)

osIsWindows[] = StringMatchQ[$System, "*Windows*"]

osIsClassicMac[] = StringMatchQ[$System, "*Macintosh*"]

osIsMacOSX[] = StringMatchQ[$System, "*Mac*X*"]

isPreemptiveKernel[] = $VersionNumber >= 5.1

hasFrontEnd[] := Head[$FrontEnd] === FrontEndObject

hasServiceFrontEnd[] := Head[MathLink`$ServiceLink] === LinkObject

hasLocalFrontEnd[] := hasFrontEnd[] && MathLink`CallFrontEnd[FrontEnd`Value["$MachineID"]] === $MachineID


(***********************************  contextIndependentOptions  **************************************)

(* This processes the names of options in a context-independent way.
*)
contextIndependentOptions[optName_Symbol, opts_List, defaults_List] :=
    First[ contextIndependentOptions[{optName}, opts, defaults] ]

contextIndependentOptions[optNames_List, opts_List, defaults_List] :=
    Module[{optNameStrings, stringifiedOptionSettings, stringifiedOptionDefaults},
        optNameStrings = (# /. x_Symbol :> SymbolName[x])& /@ optNames;
        stringifiedOptionSettings = MapAt[(# /. x_Symbol :> SymbolName[x])&, #, {1}]& /@ Flatten[{opts}];
        stringifiedOptionDefaults = MapAt[(# /. x_Symbol :> SymbolName[x])&, #, {1}]& /@ Flatten[{defaults}];
        optNameStrings /. stringifiedOptionSettings /. stringifiedOptionDefaults
    ]


End[]