//////////////////////////////////////////////////////////////////////////////////////
//
//   J/Link source code (c) 1999-2002, Wolfram Research, Inc. All rights reserved.
//
//   Use is governed by the terms of the J/Link license agreement, which can be found at
//   www.wolfram.com/solutions/mathlink/jlink.
//
//   Author: Todd Gayley
//
//////////////////////////////////////////////////////////////////////////////////////

package com.wolfram.jlink;

/**
 * Install is the class that has the main entry point when Java is launched from Mathematica
 * via InstallJava[]. It sets up definitions in Mathematica for calls into Java and starts up the
 * Reader thread, which waits for calls arriving from Mathematica. It has only one method of interest
 * to users, getStdLink(), which has been deprecated. 
 */

public class Install {

	static final int CALLJAVA              = 1;
	static final int LOADCLASS             = 2;
	static final int THROW                 = 3;
	static final int GC                    = 4;
	static final int RELEASEOBJECT         = 5;
	static final int RELEASEALLOBJECTS     = 6;
	static final int UNLOADCLASS           = 7;
	static final int VAL                   = 8;
	static final int ONLOADCLASS           = 9;
	static final int ONUNLOADCLASS         = 10;
	static final int SETCOMPLEX            = 11;
	static final int GETCOMPLEX            = 12;
	static final int REFLECT               = 13;
	static final int SHOW                  = 14;
	static final int SAMEQ                 = 15;
	static final int INSTANCEOF            = 16;
	static final int ALLOWRAGGED           = 17;
	static final int GETEXCEPTION          = 18;
	static final int CONNECTTOFE           = 19;
	static final int DISCONNECTTOFE        = 20;
	static final int PEEKCLASSES           = 21;
	static final int PEEKOBJECTS           = 22;
	static final int CLASSPATH             = 23;
	static final int ADDTOCLASSPATH        = 24;
	static final int SETUSERDIR            = 25;
	static final int ALLOWUICOMPUTATIONS   = 26;
	static final int UITHREADWAITING       = 27;
	static final int FORCEPOLLING          = 28;
	static final int YIELDTIME             = 29;
	static final int GETCONSOLE            = 30;
    static final int UILINK                = 33;
    static final int GETWINDOWID           = 34;
    static final int ADDTITLECHANGELISTENER= 35;

	/**
	 * This is the entry point called by the Mathematica function InstallJava.
	 * Users will not call this unless they are trying to manually establish a link between Java
	 * and Mathematica, instead of using the default launch behavior of InstallJava. For example,
	 * here is how you could establish a link using listen/connect modes. You might do this if you
	 * wanted to launch the Java runtime inside a debugger such as the one provided in your
	 * Java development environment:
	 * <pre>
	 * On the command line, or in your Java development environment:
	 *     java com.wolfram.jlink.Install -linkname 1234 -linkmode listen
	 *
	 * Then in Mathematica:
	 *     InstallJava[LinkConnect["1234"]]</pre>
	 *
	 * @param args
	 */

	public static void main(String[] args) {

		KernelLink ml;

		System.out.println("J/Link (tm)");
		System.out.println("Copyright (C) 1999-2005, Wolfram Research, Inc. All Rights Reserved.");
		System.out.println("www.wolfram.com");
		System.out.println("Version " + KernelLink.VERSION);
		System.out.println("");
		System.out.flush();

		if (MathLinkImpl.DEBUGLEVEL > 1) {
			for (int i = 0; i < args.length; i++)
				System.err.println(args[i]);
		}

        // OS/X uses a new system wherein the NSUIElement property is set in the Info.plist file,
        // which prevents the J/Link menu and dock icon from appearing. That makes this old stuff
        // for initializing Carbon obsolete. For compatibility, or potential problems that aris
        // with the new system, we will leave this code here, but not enable it unless the
        // com.wolfram.jlink.nogui property is set to false. This means that to restore the
        // old J/Link 2.x behavior you will have to unset the NSUIElement property and set the
        // com.wolfram.jlink.nogui property to false.
		if (Utils.isMacOSX()) {
			boolean noGUI = true;
			String noGUIProperty = null;
			try {
				noGUIProperty = System.getProperty("com.wolfram.jlink.nogui");
			} catch (SecurityException e) {}
			if (noGUIProperty != null && noGUIProperty.equals("false"))
				noGUI = false;
			// On OSX 10.1, we must initialize Carbon before we open the link. Otherwise there is a weird GUI problem
			// in which Java windows don't come to the foreground properly when they are clicked.
			// This initialization happens in the call below where we install About and Quit handlers. If we took the
			// handlers out for some reason, we would still need to init Carbon, for example by creating and disposing
			// a Frame(). Register the handlers via reflection because otherwise it's a pain compiling on other platforms.
			if (!noGUI) {
				try {
					Class mrjHandlerCls = Class.forName("com.wolfram.jlink.MRJHandlers");
					java.lang.reflect.Method setupMeth = mrjHandlerCls.getDeclaredMethod("setup", new Class[]{});
					setupMeth.invoke(null, null);
				} catch (Exception e) {}
				// The J/Link app comes to the foreground when Carbon is initialized above. Here we restore
				// the Mathematica front end as the foreground app. Manually load the library, as we are calling
				// a static method before a link has been created.
				NativeLink.loadNativeLibrary();
				NativeLink.mathematicaToFront();
			}
		}

		try {
			// This security manager only prevents calls to System.exit().
			System.setSecurityManager(JLinkSecurityManager.getInstance());
		} catch (SecurityException e) {}

		try {
			ml = MathLinkFactory.createKernelLink(args);
		} catch (MathLinkException e) {
			System.err.println("FATAL ERROR: link creation failed.");
			return;
		}

		if (!install(ml)) {
			ml.close();
			return;
		}

		if (Utils.isWindows()) {
			// Here we hide the java DOS window that is created if java.exe is the runtime used instead of javaw.exe.
			if (ml instanceof WrappedKernelLink) {
				MathLink impl = ((WrappedKernelLink) ml).getMathLink();
				if (impl instanceof NativeLink)
					NativeLink.hideJavaWindow();
			}
		}

		Reader.startReader(ml, true, false);
	}


	/**
	 * @deprecated As of J/Link 1.1, use {@link StdLink#getLink() StdLink.getLink()} instead.
	 */
	public static KernelLink getStdLink() {
		return StdLink.getLink();
	}


	public static boolean install(MathLink ml) {

		// Adding jXX functions here requires also adding the symbols to the JLink`Java.m file.
		try {
			ml.connect();
			ml.put("Begin[\"" + KernelLinkImpl.PACKAGE_PROTECTED_CONTEXT + "\"]");
			definePattern(ml, "jCallJava[indices_List, argCount_Integer, args___]", "{indices, argCount, args}", CALLJAVA);
			definePattern(ml, "jLoadClass[class_String, objSupplyingClassLoader_Symbol, isComplexClass_]", "{class, objSupplyingClassLoader, isComplexClass}", LOADCLASS);
			definePattern(ml, "jThrow[exc_, msg_String]", "{exc, msg}", THROW);
			definePattern(ml, "jGC[]", "{}", GC);
			definePattern(ml, "jReleaseObject[instances:{__Symbol}]", "{instances}", RELEASEOBJECT);
			definePattern(ml, "jReleaseAllObjects[classIndex_Integer]", "{classIndex}", RELEASEALLOBJECTS);
			definePattern(ml, "jUnloadClass[classIndex_Integer]", "{classIndex}", UNLOADCLASS);
			definePattern(ml, "jVal[instance_Symbol]", "{instance}", VAL);
			definePattern(ml, "jOnLoadClass[classIndex_Integer]", "{classIndex}", ONLOADCLASS);
			definePattern(ml, "jOnUnloadClass[classIndex_Integer]", "{classIndex}", ONUNLOADCLASS);
			definePattern(ml, "jSetComplex[classIndex_Integer]", "{classIndex}", SETCOMPLEX);
			definePattern(ml, "jGetComplex[]", "{}", GETCOMPLEX);
			definePattern(ml, "jReflect[classIndex_Integer, type_Integer, inherited:(True | False)]", "{classIndex, type, inherited}", REFLECT);
			definePattern(ml, "jShow[wnd_Symbol]", "{wnd}", SHOW);
			definePattern(ml, "jSameQ[obj1_Symbol, obj2_Symbol]", "{obj1, obj2}", SAMEQ);
			definePattern(ml, "jInstanceOf[obj_Symbol, className_String]", "{obj, className}", INSTANCEOF);
			definePattern(ml, "jAllowRaggedArrays[allow:(True | False)]", "{allow}", ALLOWRAGGED);
			definePattern(ml, "jGetException[]", "{}", GETEXCEPTION);
			definePattern(ml, "jConnectToFEServer[linkName_String, protocol_String]", "{linkName, protocol}", CONNECTTOFE);
			definePattern(ml, "jDisconnectToFEServer[]", "{}", DISCONNECTTOFE);
			definePattern(ml, "jPeekClasses[]", "{}", PEEKCLASSES);
			definePattern(ml, "jPeekObjects[]", "{}", PEEKOBJECTS);
			definePattern(ml, "jClassPath[]", "{}", CLASSPATH);
			definePattern(ml, "jAddToClassPath[dirs:{__String}, searchForJars:(True | False)]", "{dirs, searchForJars}", ADDTOCLASSPATH);
			definePattern(ml, "jSetUserDir[dir_String]", "{dir}", SETUSERDIR);
			definePattern(ml, "jUIThreadWaitingQ[]", "{}", UITHREADWAITING);
			definePattern(ml, "jAllowUIComputations[allow:(True | False), enteringModal:(True | False):False]", "{allow, enteringModal}", ALLOWUICOMPUTATIONS);
			definePattern(ml, "jForcePolling[force:(True | False), yielder:(True | False)]", "{force, yielder}", FORCEPOLLING);
			definePattern(ml, "jYieldTime[millis_Integer]", "{millis}", YIELDTIME);
			definePattern(ml, "jGetConsole[]", "{}", GETCONSOLE);
            definePattern(ml, "jUILink[name_String, prot_String, useSnooper:(True | False):False]", "{name, prot, useSnooper}", UILINK);
            definePattern(ml, "jGetWindowID[obj_Symbol]", "{obj}", GETWINDOWID);
            definePattern(ml, "jAddTitleChangeListener[obj_Symbol, func_String]", "{obj, func}", ADDTITLECHANGELISTENER);
			ml.put("End[]");
			ml.putSymbol("End");
			ml.flush();
			return true;
		} catch (MathLinkException e) {
			if (MathLinkImpl.DEBUGLEVEL > 1) System.err.println("Fatal error: MathLinkException during Install.");
			return false;
		}
	}


	// Analogous to the C function created by mprep for C installable MathLink programs. Note that the built-in DefineExternal
	// function in Mathematica is not abort-safe, so J/Link uses its own version, jlinkDefineExternal.
	private static void definePattern(MathLink ml, String patt, String args, int index) throws MathLinkException {

		ml.putFunction(KernelLinkImpl.PACKAGE_PROTECTED_CONTEXT + "jlinkDefineExternal", 3);
		ml.put(patt);
		ml.put(args);
		ml.put(index);
	}

}
