//////////////////////////////////////////////////////////////////////////////////////
//
//   J/Link source code (c) 1999-2002, Wolfram Research, Inc. All rights reserved.
//
//   Use is governed by the terms of the J/Link license agreement, which can be found at
//   www.wolfram.com/solutions/mathlink/jlink.
//
//   Author: Todd Gayley
//
//////////////////////////////////////////////////////////////////////////////////////

package com.wolfram.jlink;

import java.util.Hashtable;
import java.util.Stack;


/**
 * StdLink contains two methods that you use to interact with the link back to the kernel
 * in cases where Java is "installed" into Mathematica (that is, InstallJava[] has been called
 * in the Mathematica session). The getLink() method returns this link, and requestTransaction()
 * must be called from the user-interface thread before calling into Mathematica.
 * <p>
 * You never create or use an instance of StdLink; it simply is a container for some methods
 * and state related to the link back to the kernel. The name is inspired by the 'stdlink' global
 * variable that holds the link in C-language "installable" MathLink programs generated by the mprep tool.
 */

public class StdLink {
	
	// These are the values that can be taken by allowUIComputations
	private static final int NONE			= 0;
	private static final int JUST_ONE	= 1;
	private static final int MODAL		= 2;
	
	// mainLink is the link that the Reader thread was started with.
	private static KernelLink mainLink;
    // All UI-triggered computations are sent to this link.
    private static KernelLink uiLink;
	private static boolean mainLinkHasReader = false;
	private static boolean allowRaggedArrays = false;
	private static int allowUIComputations = NONE;
	private static boolean lastPktWasAllowUIComps = false;
	private static boolean forcePoll = false;
	private static boolean uiPermissionRequested = false;
	private static Hashtable stdLinkHash = new Hashtable(4, 0.75F);
	private static Object stdLinkLock = new Object();
	private static Object uiLock = new Object();

	
	// A Stack is used to handle recursive calls to handleCallPacket. Each thread is associated
	// with a stack of KernelLinks representing each time the handleCallPacket function is entered
	// (setup() is called at this time). When we leave handleCallPacket we call remove(),
	// which pops from this stack, and then removes a thread record from the
	// hashtable only when all recursive calls have unwound. Of course, all this is somewhat
	// unlikely. Recursive calls aren't all that unlikely, but those could be handled by associating
	// a simple count with a thread-link pair. We need a stack of potentially different links on each
	// thread only to handle the exceedingly rare cases when a call into Java would trigger a call
	// to another kernel, which would then in turn call back into Java.
	// This scheme is absurdly complicated for virtually all conceivable uses. But it works,
	// and it's safe, so I might as well leave it.
	
	/**
	 * When called during a session when Java is "installed" into Mathematica (i.e., InstallJava[] has
	 * been called in Mathematica), getLink() returns the KernelLink that points back to Mathematica.
	 * It returns null if Java is not being used from Mathematica. Java methods need to obtain the link
	 * back to Mathematica in a number of circumstances, such as if they want to return a result to Mathematica
	 * manually, or trigger computations in Mathematica before they return, or if they are called
	 * from the user-interface thread in response to a user action like clicking a button.
	 * <p>
	 * Here is an example of how it might be called in a method that wants to return a result to Mathematica
	 * manually, instead of having its normal return value sent back.
	 * <pre>
	 * 	KernelLink link = StdLink.getLink();
	 * 	if (link != null) {
	 * 		link.beginManual();
	 * 		... code here to put a result to Mathematica
	 * 	}</pre>
	 */
	
	public static KernelLink getLink() {
		
		synchronized (stdLinkLock) {
			Object key = Thread.currentThread();
			Stack s = (Stack) stdLinkHash.get(key);
			if (s != null && !s.empty()) {
				// This call is from code that is itself being called from Mathematica (handleCallPacket is on the stack).
				return (KernelLink) s.peek();
			} else if (uiLink == null) {
				// Either we are in Mathematica 5.x or earlier (no UILink), or programmer has manually
				// called setLink() to specify a link to be used (typically, this is a standalone
				// Java program that has only one link to the kernel and wants getLink() to return it.)
				return mainLink;
			} else {
				// The UI link is active and this call is not being made as part of a chain that originated in Mathematica (most likely
				// it is being made on the UI thread as a result of a user action in Java). We want to return mainLink if we are in
				// the modal state, and uiLink if not (a ShareKernel situation).
				// Acquire the uiLock because it guards reads/writes of the allowUIComputations variable.
				synchronized (uiLock) {
					return allowUIComputations == MODAL ? mainLink : uiLink;
				}
			}
		}
	}

	/**
	 * Sets the link that will be returned by getLink() when the program is not in the middle
	 * of a call from Mathematica. The reason this is useful is that you might want to migrate
	 * a program that was originally written to be scripted from Mathematica into a standalone
	 * mode (where the program is the kernel's $ParentLink and the notebook front end is no
	 * longer involved). Alternatively, you might want to maintain dual-mode functionality,
	 * where a class can be instantiated and scripted from a Mathematica session but also has
	 * a main() method that lets it run standalone. In either of these circumstances, you may
	 * have code that calls StdLink.getLink() to give the link back to Mathematica. By exposing
	 * setLink(), J/Link lets you assign the "StdLink" yourself, so that you don't have to
	 * modify all the code that calls StdLink.getLink(). You are spared from having to keep
	 * track of whether your program is being scripted (and thus has a valid StdLink) or not.
	 * 
	 * @param ml
	 * @since J/Link version 1.1.2
	 */
	// Compare to the setFromReader() method, which is called from Install.startReader().
	public static void setLink(KernelLink ml) {
		mainLink = ml;
	}


	/**
	 * Must be called from code that calls into Mathematica from the user-interface thread.
	 * For each computation you want to send, you must call requestTransaction prior. The
	 * requestTransaction method will block until Mathematica is in a state ready to accept
	 * incoming evaluations from Java. Examples of code that needs to use requestTransaction are
	 * the various &quot;MathListener&quot; classes, which trigger calls into Mathematica as the result of
	 * a user action in Java (such as clicking a button or dragging a slider). Examples of code that
	 * does not need to call requestTransaction are the typical callbacks from Java methods that are
	 * themselves being called from Mathematica. In other words, you only need to use requestTransaction
	 * if the call into Mathematica <i>originates</i> from Java (like a user-interface action), not if
	 * it is part of a chain of back-and-forth calls that includes Mathematica calling into Java.
	 * <p>
	 * If you are writing a "listener"-type class (i.e., one that implements java.util.EventListener) that calls
	 * into Mathematica as a result of some user action on the user-interface thread, you should consider
	 * making your class a subclass of MathListener. The MathListener class handles all the details of
	 * the interaction with Mathematica for you, including calling requestTransaction().
	 * <p>
	 * Here is a typical example of code that calls requestTransaction(). Note that the code that sends the
	 * evaluation and reads the answer must still be in a synchronized block. This synchronization solves
	 * a different problem than requestTransaction(). Make sure you enter the synchronized block <i>after</i>
	 * requestTransaction(), or the user-interface thread will hang forever.
	 * <pre>
	 *     KernelLink ml = StdLink.getLink();
	 *     StdLink.requestTransaction();
	 *     synchronized (ml) {
	 *         try {
	 *             ml.evaluate(&quot;buttonClickFunction[]&quot;);
	 *             ml.discardAnswer();
	 *             // evaluate() sends an EvaluatePacket. You could also send the EvaluatePacket manually:
	 *             //ml.putFunction(&quot;EvaluatePacket&quot;, 1);
	 *             //ml.putFunction(&quot;ToExpression&quot;, 1);
	 *             //ml.put(&quot;buttonClickFunction[]&quot;);
	 *             //ml.discardAnswer();
	 *         } catch (MathLinkException e) {}
	 *     }</pre>
	 */
	public static void requestTransaction() {
        
 		// First simple test: bail out if there is no Reader thread running. Typical case for a standalone Java program.
		if (!mainLinkHasReader)
			return;
		
       // This is the link that the user's subsequent call to StdLink.getLink() will return. Depending on which link
       // this is, we need to do different things below.
        KernelLink linkThatWillBeUsed = getLink();
        
        // This is an error condition (the user will get a NullPointerException later when they call getLink() and
        // try to use it), but there is nothing to do here.
        if (linkThatWillBeUsed == null)
            return;
		
        // It's always safe to call on the UI link.
        if (linkThatWillBeUsed == uiLink)
            return;
        
        // Getting here means that the link that will be used is the main JavaLink[] that the Reader thread also uses.
		
		// If we are in the middle of a callback from Mathematica, handleCallPacket is on the stack, which we can
		// detect by having the thread represented in stdLinkHash. In this circumstance it is safe to call M.
		synchronized (stdLinkLock) {
			Object key = Thread.currentThread();
			Stack s = (Stack) stdLinkHash.get(key);
			if (s != null && !s.empty())
				return;
		}
	
		// If we get here it means that we will be using the main JavaLink[], and we are not in the middle of a call from M.
        // We must wait until the Reader thread frees us to proceed.
		synchronized (uiLock) {
			uiPermissionRequested = true;
			while (allowUIComputations == NONE) {
				try { uiLock.wait(); } catch (Exception e) {}
			}
			if (allowUIComputations == JUST_ONE)
				allowUIComputations = NONE;
			// This next value is not relevant for modal loops.
			uiPermissionRequested = false;
		}
	}
	

    // The separate UI link is a Mathematica 5.1 and later feature.
    public static void setUILink(KernelLink uiLink) {
        StdLink.uiLink = uiLink;
    }
    
    public static KernelLink getUILink() {
        return uiLink;
    }
    
    public static KernelLink getMainLink() {
        return mainLink;
    }
    
    
	// Marks this link as having a Reader thread. In other words, it marks the link as being a "true" StdLink.
	// Because setLink() is a public method, some of the links that users will call it with do
	// not have the properties of a StdLink. Specifically, the requestTransaction() method needs to distinguish
	// links that have a Reader thread associated with them (and thus all the issues that make requestTransaction()
	// necessary) and those that do not. With setLink() public, it no longer suffices to examine whether there
	// is a mainLink in requestTransaction(). We need to specifically know whether the mainLink is a Reader
	// link or not. Thus a new method (not public) was needed.
	static void setHasReader(boolean hasReader) {
		mainLinkHasReader = hasReader;
	}
	
	
	static void setup(KernelLink ml) {
		
		synchronized (stdLinkLock) {
			Object key = Thread.currentThread();
			Stack s = (Stack) stdLinkHash.get(key);
			if (s == null) {
				// One-time only operation for any given thread.
				s = new Stack();
				stdLinkHash.put(key, s);
			}
			s.push(ml);
		}
	}
	
	static void remove() {
		
		synchronized (stdLinkLock) {
			Object key = Thread.currentThread();
			Stack s = (Stack) stdLinkHash.get(key);
			// s can never be null.
			s.pop();
		}
	}
	
	// Calling this releases the UI thread if it is waiting, allowing it to proceed with initiating computations in
	// Mathematica. M calls this (via jAllowUIComputations[]) when it is safe. It is also called by M to signal that
	// it is no longer safe.
	static void allowUIComputations(boolean allow, boolean enteringModal) {
		
		synchronized (uiLock) {
			if (allow) {
				lastPktWasAllowUIComputations(true);
				if (enteringModal || uiPermissionRequested) {
					allowUIComputations = (enteringModal ? MODAL : JUST_ONE);
					uiLock.notify();
				}
			} else {
				// 'allow' is only false when we signal the end of the modal state. There is no Mathematica call to signal the end of
				// the single transaction state allowed by ShareKernel[]--allowUITransaction is turned off in requestTransaction.
				allowUIComputations = NONE;
			}
		}
	}
	
	// This is the equivalent of LinkReadyQ when called by ShareKernel. The Java link will never have any data on it
	// because it requires to be specifically told it's OK to send something from the UI thread. Instead, we ask whether
	// anyone is waiting for permission to send something.
	static boolean uiThreadWaiting() {
		synchronized (uiLock) {
			return uiPermissionRequested;
		}
	}
	
	// Called (indirectly) from Mathematica.
	static void forcePolling(boolean b) {
		forcePoll = b;
	}
	
	// It's ugly to have to store this state, but it would be more work to avoid it. We need this because Mathematica does
	// not always call jAllowUIComputations[False] to match every jAllowUIComputations[True]. It does this for modal loops and
	// although it doesn't do it in ServiceJava, it would be easy to add. The problem is in ShareKernel--I would have to muck
	// with the hairy code for mainLoop[]. Since Java is not always told explicitly that UI comps are no longer allowed, the
	// Reader thread must poll until the next CallPacket (which will signal the end of the single UI computation that was allowed,
	// unless we are in the modal state, which is handled separately).
	static void lastPktWasAllowUIComputations(boolean b) {
		lastPktWasAllowUIComps = b;
	}
	
	// This is what the Reader thread calls to determine whether it must use the polling vs. blocking methods for reading the link.
	// It must poll if the Mathematica side has demanded it (forcePoll), we are running a modal loop, or if the very last CallPacket
	// encountered was a call to allowUIComputations().
	static boolean mustPoll() {
		return forcePoll || lastPktWasAllowUIComps || allowUIComputations == MODAL;
	}

}
