
package com.wolfram.jlink.util;

import com.wolfram.jlink.*;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;


/**
 * MathematicaTask is an Ant task that allows you to call Mathematica from Ant build files.
 * Ant (<a href="http://ant.apache.org">http://ant.apache.org</a>) is a popular Java-based
 * build tool.
 * <p>
 * Although Ant has a wide array of tasks, it is not a scripting language. It can be quite convenient to use Mathematica as a scripting language in Ant. 
 * 
 * <h3>Parameters</h3>
 * <table border="1" cellpadding="2" cellspacing="0">
 * <tr>
 *  <td valign="top"><b>Attribute</b></td>
 *  <td valign="top"><b>Description</b></td>
 *  <td align="center" valign="top"><b>Required</b></td>
 * </tr>
 * <tr>
 *  <td valign="top">exe</td>
 *  <td valign="top">The path to the Mathematica kernel to launch</td>
 *  <td valign="top" align="center">Either exe or cmdLine must be specified</td>
 * </tr>
 * <tr>
 *  <td valign="top">cmdLine</td>
 *  <td valign="top">The full set of MathLink command-line args to use to connect to the kernel</td>
 *  <td valign="top" align="center">Either exe or cmdLine must be specified</td>
 * </tr>
 * <tr>
 *  <td valign="top">fresh</td>
 *  <td valign="top">Whether to start a fresh kernel for this task (if a previously-used
 *                     kernel is running, it will be quit first)</td>
 *  <td valign="top" align="center">No, defaults to false</td>
 * </tr>
 * <tr>
 *  <td valign="top">quit</td>
 *  <td valign="top">Whether to force the kernel to quit immediately after this task finishes</td>
 *  <td valign="top" align="center">No, defaults to false</td>
 * </tr>
 * </table>
 * <p>
 * To use the mathematica task, you must include a taskdef line like the following
 * examples your build.xml file:
 * <pre>
 * &lt;!-- If JLink.jar is on your system CLASSPATH: --&gt;
 * &lt;taskdef name="mathematica" classname="com.wolfram.jlink.util.MathematicaTask"/&gt;
 *
 * &lt;!-- If JLink.jar is NOT on your system CLASSPATH: --&gt;
 * &lt;taskdef name="mathematica" classname="com.wolfram.jlink.util.MathematicaTask"
 *           classpath="/full/path/to/JLink.jar"/&gt; </pre>
 * You have two ways of specifying how to launch or connect to the Mathematica kernel. Most
 * users will use the exe parameter, which specifies the path to the kernel to launch. In rare cases
 * you might need more control over how the kernel is launched, or you might want to connect to
 * an already-running kernel (not launched by Ant). In such cases, you can use the cmdLine
 * parameter, which lets you specify a full set of command-line MathLink arguments (e.g.,
 * "-linkmode connect -linkname 1234 -linkprotocol tcpip").
 * <p>
 * The examples below assume that you have a mathExe property defined that gives the path to
 * the kernel to launch. Here are typical definitions of that property for various operating systems:
 * <pre>
 * &lt;!-- Windows --&gt;
 * &lt;property name="mathExe", value="c:/program files/wolfram research/mathematica/5.2/mathkernel"/&gt;
 * 
 * &lt;!-- Linux, Unix ('math' is typically on PATH) --&gt;
 * &lt;property name="mathExe", value="math"/&gt;
 * 
 * &lt;!-- Mac OS/X --&gt;
 * &lt;property name="mathExe", value="/Applications/Mathematica\ 5.2.app/Contents/MacOS/MathKernel"/&gt;</pre>
 * 
 * Here is an example...
 * 
 * <pre>
 *    &lt;target name="mathExample"&gt;
 *      &lt;mathematica exe="${mathExe}"&gt;   
 *        &lt;![CDATA[
 *           For[i = 0, i < 10, i++,
 *              echo = Ant["project"]@createTask["echo"];
 *              echo@setMessage[ToString[i^2]];
 *              echo@perform[]   
 *           ]   
 *        ]]&gt;
 *      &lt;/mathematica&gt;   
 *    &lt;/target&gt; </pre>
 *
 * <pre>
 *    &lt;target name="mathExample"&gt;
 *      &lt;mathematica exe="${mathExe}"&gt;   
 *        &lt;![CDATA[
 *           For[i = 0, i < 10, i++,
 *              AntTask["echo", "message"->ToString[i^2]]
 *           ]   
 *        ]]&gt;
 *      &lt;/mathematica&gt;   
 *    &lt;/target&gt; </pre>
 *
 * @since 3.1
 */

public class MathematicaTask extends Task implements PacketListener {

    static KernelLink ml = null;
    
    String exe = null;
    String cmdLine = null;
    
    boolean freshKernel = false;
    boolean quit = false;
    String runFile = null;
    String code = "";
    String failMsg = null;
    
    
    public void setExe(String exe) {
        this.exe = exe;
    }

    public void setCmdLine(String cmdLine) {
        this.cmdLine = cmdLine;
    }

    public void setFresh(boolean fresh) {
        this.freshKernel = fresh;
    }
    
    public void setQuit(boolean quit) {
        this.quit = quit;
    }
    
    public void setRunFile(String runFile) {
        this.runFile = runFile;
    }
    
    
    // Called by Ant with the text of the CDATA section, which is the code to execute.
    //
    public void addText(String code) {
        this.code = code;
    }

    
    // Called from Mathematica via the AntFail function to force a build failure.
    // 
    public void setFail(String msg) {
        failMsg = msg;
    }
    
    
    public void execute() throws BuildException {
    
        if (freshKernel)
            closeKernel();
            
        if (ml == null)
            ml = initKernel();
            
        try {
            ml.putFunction("EvaluatePacket", 1);
            ml.putFunction("Set", 2);
              ml.putSymbol("$this");
              ml.put(this);
            ml.discardAnswer();
          
            if (runFile != null) {
                ml.putFunction("EvaluatePacket", 1);
                ml.putFunction("Get", 1);
                ml.put(runFile);
                ml.discardAnswer();
            }
            if (code != null && !code.equals("")) {
                ml.evaluate(code);
                ml.discardAnswer();
            }
            if (failMsg != null)
                throw new BuildException(failMsg, getLocation());
        } catch (MathLinkException e) {
            throw new BuildException("Link error: " + ml.errorMessage(), getLocation());
        } finally {
            failMsg = null;
            if (quit || ml.error() != MathLink.MLEOK)
                closeKernel();
        }
    }
    
    
    protected KernelLink initKernel() {
    
        KernelLink ml = null;
        String[] args = new String[]{"-linkmode", "launch", "-linkname", ""};
        boolean useArray = false;
        
        if (exe != null) {
            useArray = true;
            String quoteChar = Utils.isWindows() ? "" : "'";
            args[3] = quoteChar + exe + quoteChar + " -mathlink";
        } else if (cmdLine == null) {
            throw new BuildException("Must specify exe or cmdLine attribute to control kernel launch.", getLocation());
        }
            
        try {
            if (useArray) {
                // Simple kernel launch.
                ml = MathLinkFactory.createKernelLink(args);
                ml.discardAnswer();
            } else {
                ml = MathLinkFactory.createKernelLink(cmdLine);
                ml.connect();
            }
            ml.enableObjectReferences();
            ml.evaluateToInputForm(startupCode, 0);
            ml.addPacketListener(this);
        } catch (MathLinkException e) {
            throw new BuildException("Failed to launch or connect to Mathematica kernel: " + e.getMessage(), getLocation());
        }
        return ml;

    }
    
    
    protected void closeKernel() {
        
        if (ml!= null) {
            ml.terminateKernel();
            ml.close();
            ml = null;
        }
    }
    
    // Empty Javadoc comment to avoid importing inherited one.
    /** &nbsp;
    */
    public boolean packetArrived(PacketArrivedEvent evt) throws MathLinkException {
         if (evt.getPktType() == MathLink.TEXTPKT) {
             KernelLink ml = (KernelLink) evt.getSource();
             log(ml.getString()); 
         }
         return true;
     }
     
     
    private static String startupCode =
        "Ant[obj_String] :=                                                     " +
        "   Switch[ToLowerCase[obj],                                            " +
        "       \"project\",                                                    " +
        "           Ant[\"target\"]@getProject[],                               " +
        "       \"target\",                                                     " +
        "           $this@getOwningTarget[],                                    " +
        "       \"task\",                                                       " +
        "           $this,                                                      " +
        "       \"location\",                                                   " +
        "           $this@getLocation[],                                        " +
        "       _,                                                              " +
        "           AntLog[\"Unknown object type in Ant function: \" <> obj];   " +
        "           $Failed                                                     " +
        "   ];                                                                  " +
        "AntTask[name_String, args___?OptionQ] :=                               " +
        "   JavaBlock[                                                          " +
        "      Module[{task, attrNames, attrVals},                              " +
        "          task = Ant[\"project\"]@createTask[name];                    " +
        "          attrNames =                                                  " +
        "             StringReplace[#, a_ ~~ b___  :> ToUpperCase[a] <> b]& /@  " +
        "                 First /@ Flatten[{args}];                             " +
        "          attrVals = Last /@ Flatten[{args}];                          " +
        "          With[{meth = ToExpression[\"set\" <> #1]},                   " +
        "              task@meth[#2]                                            " +
        "          ]& @@@ Thread[{attrNames, attrVals}];                        " +
        "          task@perform[]                                               " +
        "      ]                                                                " +
        "   ];                                                                  " +
        "AntLog[msg_String] := Ant[\"project\"]@log[msg];                       " +
        "AntLog[e_] := AntLog[ToString[e, FormatType->InputForm]];              " +
        "AntProperty[p_String] := Ant[\"project\"]@getProperty[p];              " +
        "AntSetProperty[p_String, val_String] := Ant[\"project\"]@setProperty[p, val];" +
        "AntFail[msg_String] := ($this@setFail[msg]; Abort[])";

}
