
(*******************************************)
BeginPackage["NumericalMath`EquationTrekker`", {"GUIKit`", "Utilities`FilterOptions`"}];
(*******************************************)

(*******************************
   Usage Messages
 *******************************)
 
EquationTrekker::usage = "EquationTrekker[eqn, {x,y}, {t, tmin, tmax}] brings up a window and then plots the solution of the second order differential equation, eqn, such that {x[0], y[0]} is the point at the mouse cursor. EquationTrekker[eqn, x, {t, tmin, tmax}] brings up a window and then plots the solution of the first order differential equation, eqn, such that {t0, x[t0]} is the point at the mouse cursor. If you drag the mouse with the button down, the solution is updated continuously. If the equation is a second order equation, x' is considered as the y variable.";

EquationTrekkerNonModal::usage = "EquationTrekkerNonModal is the non-modal dialog version of EquationTrekker.";

Parameter::usage = "Parameter[name, value] specifies a variable which can be modified interactively.";

TrekParameters::usage = "TrekParameters is an option to EquationTrekker specifying the dynamic parameter objects.";
TrekGenerator::usage = "TrekGenerator is an option to EquationTrekker specifying the system of trek generation.";

InitializeGenerator::usage = "InitializeGenerator is a function called to initialize a generator for trek points.  To define a generator with name gen, you should define rules so that InitializeGenerator[gen, eqns, vars, {ivar, begin, end}] returns gen[data], where data includes whatever data is needed to generate trek points.";

DifferentialEquationTrek::usage = "DifferentialEquationTrek is a value for the option TrekGenerator which shows the solution of differential equations.";

DifferentialEquationStrobe::usage = "DifferentialEquationStrobe is a value for the option TrekGenerator which shows the solution of differential equations at equally spaced values of the independent variable.  Some Poincare sections can be made with this generator.";

FindMinimumTrek::usage = "FindMinimumTrek is a value for the option TrekGenerator.";

EquationTrekkerState::usage = "EquationTrekkerState[data] contains sufficient data so that if you use EquationTrekker[EquationTrekkerState[data]], the trek window will be restored to the point at which the state was saved.";

TrekData::usage = "TrekData[data] represents the data which is shown as a single path or set of points in the EquationTrekker.  By default, only the initial conditions are shown."

(*******************************)
Begin["`Private`"]
(*******************************)

(*******************************
   Options
 *******************************)

Options[EquationTrekker] = {
  PlotRange -> {Automatic,{-1,1}},
  ImageSize -> {400,400}, 
  TrekParameters -> {}, 
  TrekGenerator -> DifferentialEquationTrek
  };

Options[DifferentialEquationTrek] = Options[NDSolve];
Options[DifferentialEquationStrobe] = Prepend[Options[NDSolve], "Frequency"->1];

SetAttributes[ProcessOptionNames, Listable]; 
ProcessOptionNames[(r : (Rule | RuleDelayed))[name_Symbol, val_]] := 
    r[SymbolName[name], val];
ProcessOptionNames[opt_] := opt;

(*******************************
   Messages
 *******************************)
 
EquationTrekker::ncsol = "Could not compute the solution `1`";
EquationTrekker::prange = "Value of option PlotRange->`1` should be in the form {{xmin, xmax},{ymin, ymax}}.";
EquationTrekker::isize = "Value of option ImageSize->`1` should be in the form {horizontal pixels, vertical pixels}.";
EquationTrekker::tgen = "Value of option TrekGenerator->`1` should be the name of a trek generator or a list with a name followed by generator options."

(* What should these say? *)
DifferentialEquationTrek::only = "EquationTrekker for differential equations is for a single first order equation, single second order equation, or two first order equations.";
DifferentialEquationStrobe::only = "EquationTrekker for differential equations with strobe frequency is for a single second order equation, or two first order equations.";
DifferentialEquationStrobe::freal = "The value of the option Frequency->`1` should be a positive real number."
FindMinimumTrek::only = "EquationTrekker for minimization is only for two variable functions.";
  
(*******************************
   EquationTrekkerState Formatting

   The main idea here is to format invidual parts of the state
   expression.  This is one way to leave the graphics alone so
   that it shows appropriately in all versions.

 *******************************)

(*
    TrekData:  The idea here is to keep the data inside, but have it only 
               show the conditions in the formatting.
*)

TrekDataFormat[TrekData[disp_, cond_, ivdata_, {color_, style_}], form_] := 
    StringJoin[
        "TrekData[\"",
        ToString[disp, form],
        "\", \"<>\"]"];

Format[tdata_TrekData, OutputForm] := TrekDataFormat[tdata, OutputForm];

Format[tdata_TrekData, TextForm] := TrekDataFormat[tdata, TextForm];

TrekData /: MakeBoxes[
     tdata:TrekData[disp_, cond_, ivdata_, {color_, style_}], form_] :=
  InterpretationBox[StyleBox[#, "FontColor"->color], tdata]& [
        TrekDataFormat[tdata, form]];

EquationTrekkerStateFormat[EquationTrekkerState[indata_, parms_, trekdata_, opts_], form_] := 
    StringJoin[
        "EquationTrekkerState[\"", 
        ToString[indata, form], "\",\"",
        ToString[parms, form], "\",",
        ToString[TableForm[trekdata], form], ",\" <>\"]"];

Format[ets_EquationTrekkerState, OutputForm] := 
    EquationTrekkerStateFormat[ets, OutputForm]

Format[ets_EquationTrekkerState, TextForm] := 
    EquationTrekkerStateFormat[ets, TextForm]

EquationTrekkerState /: MakeBoxes[ets_EquationTrekkerState, form_] := 
    InterpretationBox[#, ets]&[EquationTrekkerStateFormat[ets, form]]

(*******************************
   EquationTrekker
 *******************************)
 
EquationTrekker[eqn_, dvars_, None, opts___] := 
  EquationTrekker[eqn, dvars, {None, 0, 1}, opts];

EquationTrekker[eqn_, dvars_, {ivar_, begin_, end_}, opts___] := 
  GUIRunModal["EquationTrekker/TrekFrame", {eqn, dvars, {ivar, begin, end}, opts}];

EquationTrekker[state:EquationTrekkerState[{eqns_, dvars_, iv_}, __]] := 
  EquationTrekker[eqns, dvars, iv, "State" -> state];


EquationTrekkerNonModal[eqn_, dvars_, None, opts___] := 
  EquationTrekkerNonModal[eqn, dvars, {None, 0, 1}, opts];

EquationTrekkerNonModal[eqn_, dvars_, {ivar_, begin_, end_}, opts___] := 
  GUIRun["EquationTrekker/TrekFrame", {eqn, dvars, {ivar, begin, end}, opts}];

EquationTrekkerNonModal[state:EquationTrekkerState[{eqns_, dvars_, iv_}, __]] := 
  EquationTrekkerNonModal[eqns, dvars, iv, "State" -> state];

(*****************************************
    DifferentialEquationTrek - TrekGenerator
 *****************************************)
 
DifferentialEquationTrek /: InitializeGenerator[DifferentialEquationTrek, eqns_, 
  dvarsin_, {ivar_, begin_, end_}, opts___] := 
Module[{dvars = Flatten[{dvarsin}]},
    (* Check form of dependent variables to be sure we have at most
       two or a first or second order equation.  *)
    If[Length[dvars] == 0 || Length[dvars] > 2,
        Message[DifferentialEquationTrek::only];
        Throw[$Failed]
    ];
    (* Convert from x[t] to x *)
    dvars = Map[If[MatchQ[#, _[ivar]], Head[#], #]&, dvars];
     
    (* Determine order and set up initial condition function *)
    order = Max[Cases[eqns, Derivative[j_][v_ /; MemberQ[dvars, v]][ivar] -> j, Infinity]];
    If[order > 2,
        Message[DifferentialEquationTrek::only];
        Throw[$Failed]
    ];

    If[Length[dvars] == 1,
        If[order == 1, 
            finit = Function[dv[#1] == First[#2]] /. dv->First[dvars],
            dvars = {First[dvars], Derivative[1][First[dvars]]}
        ];
    ];
    If[Length[dvars] == 2,
        finit = Function[Thread[Equal[{dv1[#1], dv2[#1]}, #2]]] /. Thread[{dv1, dv2}->dvars]];

    ndopts = Flatten[{FilterOptions[NDSolve, opts]}];

    DifferentialEquationTrek[{eqns, eqns, dvars, ivar, finit, ndopts, None}]
]

DifferentialEquationTrek[{origeqns_, eqns_, dvars_, ivar_, finit_, ndopts_, state_}]["Variables"[]] := 
  {ivar, dvars}
    
DifferentialEquationTrek[{origeqns_, eqns_, dvars_, ivar_, finit_, ndopts_, state_}]["Display"[]] := 
  If[ListQ[origeqns] && (Length[origeqns] == 1), First[origeqns], origeqns]
    
DifferentialEquationTrek[___]["DisplayMode"[]] := "Line"
    
DifferentialEquationTrek[{origeqns_, eqns_, dvars_, ivar_, finit_, ndopts_, state_}]["FormatTrek"[t0_, x0_, _]] := 
  finit[t0, x0]
    
DifferentialEquationTrek[{origeqns_, eqns_, dvars_, ivar_, finit_, ndopts_, state_}]["ChangeParameters"[prules_]] := 
  DifferentialEquationTrek[{origeqns, origeqns /. prules, dvars, ivar, finit, ndopts, None}]
    
(de:DifferentialEquationTrek[{origeqns_, eqns_, dvars_, ivar_, finit_, ndopts_, state_}])["GenerateTrek"[x0_, {t0_, independMin_, independMax_}]] := 
Module[{obj = de, newstate = state, sol, times, points},
    If[newstate === None,
        newstate = NDSolve`ProcessEquations[{eqns, finit[t0, x0]},dvars, {ivar, independMin, independMax}, ndopts];
        newstate = First[newstate];
        obj[[1,-1]] = newstate;
    ];
    newstate = NDSolve`ReinitializeVector[newstate, t0, List @@ x0];
    sol; (* Hack to prevent extra copy due to LastValue *)
    NDSolve`Iterate[newstate, {independMin, independMax}];
    sol = NDSolve`ProcessSolutions[newstate];
    sol = dvars /. sol;
    times = First[sol]@"Coordinates"[];
    points = Map[(#@"ValuesOnGrid"[])&, sol];
    points = Transpose[Join[times, points]];
    {points, obj}
]
    
   
(*****************************************
    DifferentialEquationStrobe - TrekGenerator
 *****************************************)
 
DifferentialEquationStrobe /: InitializeGenerator[DifferentialEquationStrobe, eqns_, dvarsin_, {ivar_, begin_, end_}, opts___] := 
Module[{dvars = dvarsin},
    (* Check form of dependent variables to be sure we have at most
       two or a first or second order equation.  *)
    If[Not[ListQ[dvars]], dvars = {dvars}];
    If[Length[dvars] == 0 || Length[dvars] > 2,
        Message[DifferentialEquationStrobe::only];
        Throw[$Failed]
    ];
    (* Convert from x[t] to x *)
    dvars = Map[If[MatchQ[#, _[ivar]], Head[#], #]&, dvars];
     
    (* Determine order and set up initial condition function *)
    order = Max[Cases[eqns, Derivative[j_][v_ /; MemberQ[dvars, v]][ivar] -> j, Infinity]];
    If[order > 2,
        Message[DifferentialEquationStrobe::only];
        Throw[$Failed]
    ];
    If[Length[dvars] == 1,
        If[order == 1, 
            (* Not supported for first order *)
            Message[DifferentialEquationStrobe::only];
            Throw[$Failed],
            dvars = {First[dvars], Derivative[1][First[dvars]]}
        ];
    ];

    finit = Function[Thread[Equal[{dv1[#1], dv2[#1]}, #2]]] /. Thread[{dv1, dv2}->dvars];

    ndopts = Flatten[{FilterOptions[NDSolve, opts]}];
    deltat = "Frequency" /. ProcessOptionNames[opts];

    DifferentialEquationStrobe[{eqns, eqns, deltat, dvars, ivar, finit, ndopts, None}]
]

DifferentialEquationStrobe[{origeqns_, eqns_, deltat_, dvars_, ivar_, finit_, ndopts_, state_}]["Variables"[]] := 
  {ivar, dvars}
    
DifferentialEquationStrobe[{origeqns_, eqns_, deltat_, dvars_, ivar_, finit_, ndopts_, state_}]["Display"[]] := 
  If[ListQ[origeqns] && (Length[origeqns] == 1), First[origeqns], origeqns]
    
DifferentialEquationStrobe[___]["DisplayMode"[]] := "Points";
    
DifferentialEquationStrobe[{origeqns_, eqns_, deltat_, dvars_, ivar_, finit_, ndopts_, state_}]["FormatTrek"[t0_, x0_, _]] := 
  finit[t0, x0]
    
DifferentialEquationStrobe[{origeqns_, eqns_, deltat_, dvars_, ivar_, finit_, ndopts_, state_}]["ChangeParameters"[prules_]] := 
  DifferentialEquationStrobe[{origeqns, origeqns /. prules, deltat /. prules, dvars, ivar, finit, ndopts, None}]
    
(de:DifferentialEquationStrobe[{origeqns_, eqns_, deltat_, dvars_, ivar_, finit_, ndopts_, state_}])["GenerateTrek"[x0_, {t0_, independMin_, independMax_}]] := 
Module[{obj = de, newstate = state, sol, t, ts},
    If[newstate === None,
        newstate = NDSolve`ProcessEquations[{eqns, finit[t0, x0]},dvars, ivar, ndopts];
        newstate = First[newstate];
        obj[[1,-1]] = newstate;
    ];
    newstate = NDSolve`ReinitializeVector[newstate, t0, x0];
    sol; (* Hack to prevent extra copy due to LastValue *)
    If[Not[TrueQ[Positive[deltat]]],
        Message[DifferentialEquationStrobe::freal, deltat];
        Return[$Failed]
    ];
    points = Reap[
        t = t0;
        If[t > independMax, 
            t = independMax;
            Sow[GetSolutionAt[newstate, t, dvars, "Backward"]]];
        While[t > independMin, 
            t -= deltat;
            Sow[GetSolutionAt[newstate, t, dvars, "Backward"]]];
        t = t0;
        If[t < independMin, 
            t = independMin;
            Sow[GetSolutionAt[newstate, t, dvars, "Forward"]]];
        While[t < independMax, 
            t += deltat;
            Sow[GetSolutionAt[newstate, t, dvars, "Forward"]]];
    ];
    {points[[2,1]], obj}
]

SetAttributes[GetSolutionAt, HoldFirst];
GetSolutionAt[state_, t_, dvars_, direction_] := 
Module[{sol, ts},
    NDSolve`Iterate[state, t];
    sol = NDSolve`ProcessSolutions[state, direction];
    ts = state@"CurrentTime"[direction];
    Prepend[Map[#[ts]&, dvars] /. sol, t]
] 
    
  
(*****************************************
    FindMinimumTrek - TrekGenerator
 *****************************************)
 
FindMinimumTrek /: InitializeGenerator[FindMinimumTrek, f_, vars_, opts___] := 
Module[{dvars = dvarsin},
    If[Not[ListQ[vars] && (Length[vars] == 2)],
        Message[FindMinimumTrek::only];
        Throw[$Failed]
    ];

    fmopts = Flatten[{FilterOptions[FindMinimum, opts]}];
    FindMinimumTrek[{f, f, vars, fmopts}]
]
    
FindMinimumTrek[{origf_, f_, vars_, fmopts_}]["Variables"[]] := vars
    
FindMinimumTrek[{origf_, f_, vars_, fmopts_}]["Display"[]] := origf
    
FindMinimumTrek[{origf_, f_, vars_, fmopts_}]["FormatTrek"[t0_, x0_, _]] := 
  Thread[{vars, x0}]
    
FindMinimumTrek[{origf_, f_, vars_, fmopts_}]["ChangeParameters"[prules_]] := 
  FindMinimumTrek[{origf, origf /. prules, vars, fmopts}]
    
FindMinimumTrek[{origf_, f_, vars_, fmopts_}]["GenerateTrek"[x0_, {t0_, independMin_, independMax_}]] := 
Module[{points},
    points = Reap[
        FindMinimum[f, Evaluate[Thread[{vars, x0}]], StepMonitor:>Sow[vars]]];
    points[[2,1]]
]


(*******************************)
End[]   (* end private context *)
(*******************************)

(*******************************)
EndPackage[];
(*******************************)