(* :Title: Unconstrained Problems *)

(*
    Small suite of unconstrained local optimization problems 
    adapted from

    "Testing Unconstrained Optimization Software" 
        by More, Garbow, and Hillstrom.

    The package includes problem specifications, solutions, and
    functions for testing preformance parameters as well as 
    functions for plotting 1 and 2-d search paths for
    FindMinimum and FindRoot.
*)

(* :Context: Optimization`UnconstrainedProblems` *)

(* :Name: Optimization`UnconstrainedProblems` *)

(* :Copyright: Copyright 1991-2003, Wolfram Research, Inc. *)

(* :Mathematica Version: 5.0 *)

BeginPackage[ "Optimization`UnconstrainedProblems`" , "Utilities`FilterOptions`"]

UnconstrainedProblem

FindRootProblem

$FindRootProblems

FindMinimumProblem

$FindMinimumProblems

GetFindRootProblem

GetFindMinimumProblem

Rosenbrock

FreudensteinRoth

PowellBadlyScaled

BrownBadlyScaled

Beale

JennrichSampson

HelicalValley

Bard

Gauss

Meyer

Gulf

Box3D

PowellSingular

Wood

KowalikOsborne

BrownDennis

Osborne1

BiggsExp6

Osborne2

Watson

ExtendedRosenbrock

ExtendedPowell

PenaltyFunctionI

PenaltyFunctionII

VariablyDimensionedFunction

TrigonometricFunction

BrownAlmostLinear

DiscreteBoundaryValue

DiscreteIntegralEquation

BroydenTridiagonal

BroydenBanded

LinearFullRank

LinearRank1

LinearRank1Z

Chebyquad

$UnconstrainedProblems

X

ProblemSolve

ProblemTime

ProblemTest

ProblemStatistics

FindMinimumPlot

FindRootPlot

Residual

Objective

Begin[ "`Private`"]

xl[x_, n_] := Array[x, n];

VQ[X_, n_] := VectorQ[X] && (Length[X] == n);

Condition[Objective[prob_, {n_, m_}][X_], VQ[X, n]] := Module[{z = Residual[prob, {n, m}][X]}, Dot[z,z]]

Condition[Residual[Rosenbrock, {2, 2}][X_], VQ[X, 2]] := Module[{x},Set[Evaluate[xl[x, 2]], X]; {10 (x[2] - x[1]^2), 1 - x[1]}];

DefaultSize[Rosenbrock] := {2, 2};

StartingPoint[Rosenbrock, 2] := {-1.2, 1.};

Minimum[Rosenbrock, {2, 2}] := {0};

Minimizer[Rosenbrock, {2, 2}] := {{1,1}};


Condition[Residual[FreudensteinRoth, {2, 2}][X_], VQ[X, 2]] := Module[{x},Set[Evaluate[xl[x, 2]], X]; {-13+x[1]+((5-x[2])*x[2]-2)*x[2], 
                               -29+x[1]+((x[2]+1)*x[2]-14)*x[2]}];

DefaultSize[FreudensteinRoth] := {2, 2};

StartingPoint[FreudensteinRoth, 2] := {0.5, -2.};

Minimum[FreudensteinRoth, {2, 2}] := {0, 48.9842536792400211999899780596695991352543420189871724667144017030744114897752195699728432533295730869423615140950959`100.};

Minimizer[FreudensteinRoth, {2, 2}] := {{5, 4}, {11.4127789869020939272458265152740449592156955287843504618590641453106338229956803713571792052512758195272054711179885`100., -0.8968052532744765181885433711814887601960761178039123845352339636723415442510799071607051986871810451181986322205029`100.}}

Condition[Residual[PowellBadlyScaled, {2, 2}][X_], VQ[X, 2]] := Module[{x},Set[Evaluate[xl[x, 2]], X]; {10^4*x[1]*x[2]-1, Exp[-x[1]]+Exp[-x[2]]-10001/10000}];

DefaultSize[PowellBadlyScaled] := {2, 2};

StartingPoint[PowellBadlyScaled, 2] := {0.,1.};

Minimum[PowellBadlyScaled, {2, 2}] := {0};

Minimizer[PowellBadlyScaled, {2, 2}] := {{0.0000109815932969981745568376164562524656719659673979784977854763420884211007923022345952840943016330245724953310057`100., 9.1061467398665240109467104903197269987365383534464844269576800022863837538853940706923936251862423194872787383719477`100.}};


Condition[Residual[BrownBadlyScaled, {2, 3}][X_], VQ[X, 2]] := Module[{x},Set[Evaluate[xl[x, 2]], X]; {x[1]-10^6,x[2]-(2*^-6),x[1]*x[2]-2}];

DefaultSize[BrownBadlyScaled] := {2, 3};

StartingPoint[BrownBadlyScaled, 2] := {1.,1.};

Minimum[BrownBadlyScaled, {2, 3}] := {0};

Minimizer[BrownBadlyScaled, {2, 3}] := {{10^6, 2*^-6}};


Condition[Residual[Beale, {2, 3}][X_], VQ[X, 2]] := Module[{x},Set[Evaluate[xl[x, 2]], X]; {3/2 - x[1]*(1 - x[2]), 9/4 - x[1]*(1 - x[2]^2), 21/8 - x[1]*(1 - x[2]^3)}];

DefaultSize[Beale] := {2, 3};

StartingPoint[Beale, 2] := {1.,1.};

Minimum[Beale, {2, 3}] := {0};

Minimizer[Beale, {2, 3}] := {{3, 1/2}};


Condition[Residual[JennrichSampson, {2, m_}][X_], VQ[X, 2]] := Module[{x},Set[Evaluate[xl[x, 2]], X]; Table[2 + 2*i - (Exp[i*x[1]] + Exp[i*x[2]]),{i,1,m}]]

DefaultSize[JennrichSampson] = {2, 10};

StartingPoint[JennrichSampson, 2] := {0.3, 0.4};

Minimum[JennrichSampson, {2, 10}] := {124.3621823556148535149855168241758279072024166421244882204521853934324555231234947370347935401588406668588782907152243`100.};

Minimizer[JennrichSampson, {2, 10}] := {{0.2578252136703640768521891757247255439033944763950168806901542815947231506628611331493825509010339478502028424378886`100., 0.2578252136703640768521891757247255439033944763950168806901542815947231506628611331493825509010339478502028424378886`100.}};


Condition[Residual[HelicalValley, {3, 3}][X_], VQ[X, 3]] := Module[{x},Set[Evaluate[xl[x, 3]], X]; {10*(x[3] - 5*ArcTan[x[1], x[2]]/Pi), 10*(Sqrt[x[1]^2 + x[2]^2] - 1), x[3]}];

DefaultSize[HelicalValley] = {3, 3};

StartingPoint[HelicalValley, 3] = {-1.,0.,0.};

Minimum[HelicalValley, {3, 3}] = {0};

Minimizer[HelicalValley, {3, 3}] = {{1,0,0}};

Condition[Residual[Bard, {3, 15}][X_], VQ[X, 3]] := Module[{i, u, v, w, x},Set[Evaluate[xl[x, 3]], X];
    MapIndexed[
        Function[i = First[#2];
            u = i;
            v = 16 - i;
            w = Min[u,v];
            #1 - (x[1] + (u/(v*x[2]+ w*x[3])))],
        {7/50, 9/50, 11/50, 1/4, 29/100, 8/25, 7/20, 39/100, 37/100, 29/50, 73/100, 24/25, 67/50, 21/10, 439/100}]    ];

DefaultSize[Bard] = {3, 15};

StartingPoint[Bard, 3] = {1., 1., 1.};

Minimum[Bard, {3, 15}] = {0.0082148773065789747519992116986671454836944034908511575690303259978237605508518314292091766681607867422057187660317`100.0, 163394/9375};

Minimizer[Bard, {3, 15}] = {{0.0824105597497889324140444684099826535110768026949726675665382954202692936749834177020519718816247573179045273912157`100., 1.1330360920297215620587184185343869398031521239137098824583636614440736434626521748698730096298047839309650970492148`100., 2.3436951786425371401417496168915405161817773629430750389577377540419958541610618726239525783942668845356434430083121`100.}, {1261/1500, -Infinity, -Infinity}}


Condition[Residual[Gauss, {3, 15}][X_], VQ[X, 3]] := Module[{x},Set[Evaluate[xl[x, 3]], X];
    MapIndexed[
        Function[x[1]*Exp[(-x[2]*((4 - First[#2]/2 - x[3])^2))/2] - #1],
        {9/10000, 11/2500, 7/400, 27/500, 259/2000, 121/500, 3521/10000, 3989/10000, 3521/10000, 121/500, 259/2000, 27/500, 7/400, 11/2500, 9/10000}]]


DefaultSize[Gauss] = {3, 15};

StartingPoint[Gauss, 3] = {.4, 1., 0.};

Minimum[Gauss, {3, 15}] = {1.127932769618648032722249267277822368401565702558571626843967208961642873661211930628395419008128080102034497173397355`100.*^-8};

Minimizer[Gauss, {3, 15}] = {{0.3989561378387566802539739665694540554747847823264461667363394486989880577736871159853806954369194703171108751503747`100.,1.0000190844878056964977925337215614338292398782353317581518728959501001682965402151023844054015956228289277593243748`100.,0}};

Condition[Residual[Meyer, {3, 16}][X_], VQ[X, 3]] := Module[{x},Set[Evaluate[xl[x, 3]], X];
    MapIndexed[
        Function[x[1]*Exp[x[2]/(45 + 5 First[#2] + x[3])] - #1],
        {34780, 28610, 23650, 19630, 16370, 13720, 11540, 9744, 8261, 7030, 6005, 5147, 4427, 3820, 3307, 2872}]];
      
DefaultSize[Meyer] = {3, 16};

StartingPoint[Meyer, 3] = {0.02, 4000., 250.};

Minimum[Meyer, {3, 16}] = {87.9458551708511208972086006671390809507137473167747547669887179561675812158428106157547513252101347107088077028802424`100.};

Minimizer[Meyer, {3, 16}] = {{0.0056096364710280525352547266518149467577897748937027382546314148020387379169582831274426710345618652003140624709835`100., 6181.346346286372279377225707742302495412164336701755823082786683523617577600127213577442261731307160143089157`100., 345.2236346241364959036007466593339744423370124830734959552987417727725163064139036060880222132993307110679029983461162`100.}};

Condition[Residual[Gulf, {3, m_Integer} /; And[m >= 3, m <= 100]][X_], VQ[X, 3]] := Module[{x},Set[Evaluate[xl[x, 3]], X];Table[
   Exp[-((Abs[25 + (-50 Log[i/10])^(2/3) - x[2]]^x[3])/x[1])] - i/10,
   {i, m}]];

DefaultSize[Gulf] = {3, 10};

StartingPoint[Gulf, 3] = {5., 2.5, 0.15};

Minimum[Gulf, {3, 10}] = {0};

Minimizer[Gulf, {3, 10}] = {{50, 25, 3/2}};

Condition[Residual[Box3D, {3, m_Integer}][X_], VQ[X, 3]] := Module[{x},Set[Evaluate[xl[x, 3]], X];Table[
    Exp[-i*x[1]/10] - Exp[-i*x[2]/10] - x[3]*(Exp[-i/10] - Exp[-i]),
    {i, m}]];

DefaultSize[Box3D] = {3, 10};

StartingPoint[Box3D, 3] = {0.,10.,20.};

Minimum[Box3D, {3, m_}] = {0,0,0};

Minimizer[Box3D, {3, m_}] = {{1,10,1},{1,10,-1},{x_, x_, 0}};

Condition[Residual[PowellSingular, {4, 4}][X_], VQ[X, 4]] := Module[{x},Set[Evaluate[xl[x, 4]], X];{x[1] + 10*x[2], Sqrt[5]*(x[3] - x[4]), (x[2] - 2*x[3])^2, Sqrt[10]*(x[1] - x[4])^2}];

DefaultSize[PowellSingular] = {4, 4};

StartingPoint[PowellSingular, 4] = {3., -1, 0., 1.};

Minimum[PowellSingular, {4, 4}] = {0}

Minimizer[PowellSingular, {4, 4}] = {{0,0,0,0}};

Condition[Residual[Wood, {4, 6}][X_], VQ[X, 4]] := Module[{x},Set[Evaluate[xl[x, 4]], X];{10*(-x[1]^2 + x[2]), 1 - x[1], 3*Sqrt[10]*(-x[3]^2 + x[4]), 1 - x[3], Sqrt[10]*(-2 + x[2] + x[4]), (x[2] - x[4])/Sqrt[10]}];

DefaultSize[Wood] = {4, 6};

StartingPoint[Wood, 4] = {-3., -1., -3., -1.};

Minimum[Wood, {4, 6}] = {0};

Minimizer[Wood, {4, 6}] = {{1, 1, 1, 1}};

Condition[Residual[KowalikOsborne, {4, 11}][X_], VQ[X, 4]] := Module[{x},Set[Evaluate[xl[x, 4]], X];MapThread[
    Function[#1 - x[1]*(#2*(#2 + x[2]))/(#2*(#2 + x[3]) + x[4])],
    {{1957/10000, 1947/10000, 347/2000, 4/25, 211/2500, 627/10000, 57/1250, 171/5000, 323/10000, 47/2000, 123/5000}, 
     {4, 2, 1, 1/2, 1/4, 1/6, 1/8, 1/10, 1/12, 1/14, 1/16}}]];
    
DefaultSize[KowalikOsborne] = {4, 11};

StartingPoint[KowalikOsborne, 4] = {0.25, 0.39, 0.415, 0.39};

Minimum[KowalikOsborne, {4, 11}] = {0.0003074859878056060813057008257788186963462228150989276750568568930970160027337575861556065509149337455991796137615`100.};


Minimizer[KowalikOsborne, {4, 11}] = {{0.192833452982508578945965312025771880194893309258123420231046338014826764826834812315186283526646681685637409763734`100., 
  0.190836238782629148533127845542248323051253762330531792569094021481126119095371362279632898230163032114451815946976`100., 
  0.1231172962778571208801897135039688751009525801615614135899764904913486488514179947049546895514705357433984133771818`100., 
  0.1357659899815370302444106843723778272634102528062273980183998640003838619327648147241442055492644813993393008998743`100.}};

Condition[Residual[BrownDennis, {4, m_Integer}][X_], VQ[X, 4]] := Module[{x},Set[Evaluate[xl[x, 4]], X];Table[
    (x[1] + i*x[2]/5 - Exp[i/5])^2 + (x[3] + x[4]*Sin[i/5] - Cos[i/5])^2,
    {i, m}]];

DefaultSize[BrownDennis] = {4, 20};

StartingPoint[BrownDennis, 4] = {25., 5., -5., -1.};

Minimum[BrownDennis, {4, 20}] = {85822.201626356344660615595738328734311412378911636887476814355314358166394000033148440481709791974778943553187`100.};

Minimizer[BrownDennis, {4, 20}] = {{-11.5944399047621653826142186088623216704706150513902895098361668796550552033190452077859914058264714995219362228008201`100., 13.2036300512072038213174624775429225985369057649467437249670543731256658759152821849997875410870811525457829821902975`100., -0.4034394881768595196441977938137369661483632638411099398849188633886381059648705595073561929591983401547843524436232`100., 0.2367787744557362991471155778176237456916622280906174241551080913609412304140959452440591357766327708872238666574046`100.}};


Condition[Residual[Osborne1, {5, 33}][X_], VQ[X, 5]] := Module[{x},Set[Evaluate[xl[x, 5]], X];Module[{t},
MapIndexed[
    Function[t = 10*(1 - First[#2]);
        #1 - (x[1] + x[2]*Exp[t*x[4]] + x[3]*Exp[t*x[5]])],
    {211/250, 227/250, 233/250, 117/125, 37/40, 227/250, 881/1000, 17/20, 409/500, 98/125, 751/1000, 359/500, 137/200, 329/500, 157/250, 603/1000, 29/50, 279/500, 269/500, 261/500, 253/500, 49/100, 239/500, 467/1000, 457/1000, 56/125, 219/500, 431/1000, 53/125, 21/50, 207/500, 411/1000, 203/500}]
]];

DefaultSize[Osborne1] = {5, 33};

StartingPoint[Osborne1, 5] = {0.5, 1.5, -1., 0.01, 0.02};

Minimum[Osborne1, {5, 33}] = {0.0000546489469748290641592164237796046761183976017441169474797673584341355572184344907059269837683491386365190849092`100.};

Minimizer[Osborne1, {5, 33}] = {{0.3754100521069520422124255233218400419049725743456095670645255125565819044160614684134797032355442863814426267300459`100., 1.9358469127123673620116304392007752305201183118693731065859045276007108487281756077657394115842932270785838771075663`100., -1.4646871366134231239005313885155152935297659729115764315418271133609380903516725698304920830807582874198949520618449`100., 0.0128675346400572887393178900738217950637429828748596743644574478674803021678687324645422642181798507951210625438718`100., 0.0221226996616726110650581528828418865693052928770220494311091845708894023709586509368989296927330484423705172948016`100.}};

Condition[Residual[BiggsExp6, {6, m_}][X_], VQ[X, 6]] := Module[{x},Set[Evaluate[xl[x,6]], X];
    Table[
        x[3] Exp[-i x[1]/10] - x[4] Exp[-i x[2]/10] + x[6] Exp[-i x[5]/10] - Exp[-i/10] + 5 Exp[-i] - 3 Exp[-2 i/5],
        {i, m}]];

DefaultSize[BiggsExp6] := {6, 13};

StartingPoint[BiggsExp6, 6] := {1., 2., 1., 1., 1., 1.};

Minimum[BiggsExp6, {6, m_}] = {0, 0};

Minimizer[BiggsExp6, {6, m_}] = {{1, 10, 1, 5, 4, 3}, {4, 10, 3, 5, 1, 1}};

Condition[Residual[Osborne2, {11, 65}][X_], VQ[X, 11]] := Module[{t, e, k, x},
Set[Evaluate[xl[x,11]], X];
MapIndexed[
    Function[
        t = (First[#2] - 1)/10;
        e[1] = Exp[-t*x[5]];
        e[k_] = Exp[-x[k + 4]*(t - x[k + 7])^2];
        Sum[x[k] e[k], {k,4}] - #1],
    {683/500, 1191/1000, 139/125, 1013/1000, 991/1000, 177/200, 831/1000, 847/1000, 393/500, 29/40, 373/500, 679/1000, 76/125, 131/200, 77/125, 
 303/500, 301/500, 313/500, 651/1000, 181/250, 649/1000, 649/1000, 347/500, 161/250, 78/125, 661/1000, 153/250, 279/500, 533/1000, 99/200, 1/2, 
 423/1000, 79/200, 3/8, 93/250, 391/1000, 99/250, 81/200, 107/250, 429/1000, 523/1000, 281/500, 607/1000, 653/1000, 84/125, 177/250, 633/1000, 
 167/250, 129/200, 79/125, 591/1000, 559/1000, 597/1000, 5/8, 739/1000, 71/100, 729/1000, 18/25, 159/250, 581/1000, 107/250, 73/250, 81/500, 
 49/500, 27/500}]
];

DefaultSize[Osborne2] = {11, 65};

StartingPoint[Osborne2, 11] = {1.3, 0.65, 0.65, 0.7, 0.6, 3., 5., 7., 2., 4.5, 5.5};

Minimum[Osborne2, {11, 65}] = {0.0401377362935477376334212799769039186546537745330247568478151566190145215166638575228176597988120427767630857137339`100.};

Minimizer[Osborne2, {11, 65}] = {
 {1.3099771546273005345317652311193954646099898502928369249214217958014808503709245530335418406524728780881585571661489`100., 
  .4315537946029889155965684043461023837113841043330568330890939543337073640763460012323708537980749569469552713622236`100., 
  .633661698960723995615479069530766518630500760266237761895658701234577297746996932842823215809047236745485151049751`100., 
  .5994305347859163025593488780516322135246337311195038621395484062422973226140078368233459628914188569382724708607076`100., 
  .7541832263280112769695143314770863705750033599704328354438396974409573158771626053958593009479414819385699623788531`100., 
  .904288579859633657767972318540329209401952026239907085227399065938279598293453901797116727981287901924182188633043`100., 
  1.3658118352370285204657457177119089622223801623371126882918365675923648548471344622880161417199372034242845085109885`100., 
  4.823698817227155581644541662031294598568219359849107836429006018068572741518280761548573247071391090812852785061069`100., 
  2.3986848661317545512805214493434838185516948080626465897817341597680627435159129504078041403749343267067591534930154`100., 
  4.5688745976676715843109273587298833819334246435427863631790238227243925518619550096604108933757479753818011217530461`100., 
  5.6753414705806411835409636383780800105543885751278442773257586142899696703802379227778151409352676091780577490722457`100.}};

Condition[Residual[Watson, {n_Integer, 31} /; And[n >= 2, n <= 31]][X_], VQ[X, n]] :=
Module[{x},Set[Evaluate[xl[x,n]], X];
Join[
    Table[
        Sum[(j - 1) x[j] (i/29)^(j - 2),{j,2,n}] - 
            Sum[x[j] (i/29)^(j - 1),{j,1,n}]^2 - 1,
        {i, 29}],
    {x[1], x[2] - x[1]^2 - 1}]];
        
DefaultSize[Watson] = {12, 31};

StartingPoint[Watson, n_] := Table[0, {n}];

Minimum[Watson, {12, 31}] = {4.7223811058917573630203382256774904762506933797132575464288543396095985148163553636314016581384386285605675286279155`100.*^-10};

Minimizer[Watson, {12, 31}] = {
 {-6.63806046452068700859450053097647253109166010681040917904331012295639539383590396026581243402532368070197774597410244`100.*^-9, 
  1.0000016441178620046500366510500172685353912396176286223798458220990062226001021294508483956600887669029976475586365`100., 
  -0.0005639322103423058497962497362264535024305741930619157194448681861439259123095965435008385891263356334541516075291`100., 
  0.3478205405040424137432769772352761011122856419097156581518251859293978269033447051594908947437814955624040347813212`100., 
  -0.1567315040961651227907065422152340526039040453879193085832716785865258035412695470375380963422736551857522890830692`100., 
  1.052815177207177751321825444986605528809188074028461740043678365532099619114670274165466965678837596222623152647223`100., 
  -3.2472711534596676980744457079458422122845358774854243043772793892777742301549119201154560396061795232240604434931266`100., 
  7.2884348979380610090202998238475605023262691573161422631499851493814924493674714755938997768458919331415114113919832`100., 
  -10.2718482413217241478223688423769503476949259971630802476794905777935791892580741495399684123387180153508289028407142`100., 
  9.0741136470753156780744234680638191800902862438109485386410243134885152326400388024526209642880207616906070737289787`100., 
  -4.5413754666925119705494842443214229370206682201437683697754400464761123624557536865483407628365691634741815415366033`100., 
  1.0120118885528104077979161693400336512766466710922573026792399131489182304976622129712231273089761059456431634525562`100.}};

Condition[Residual[ExtendedRosenbrock, {n_, n_} /; EvenQ[n]][X_], VQ[X, n]] := Module[{x},Set[Evaluate[xl[x,n]], X];
Flatten[
    Table[{10 (x[ii] - x[ii - 1]^2), 1 - x[ii - 1]},
        {ii, 2, n, 2}]
]];

DefaultSize[ExtendedRosenbrock] = {100, 100};

StartingPoint[ExtendedRosenbrock, n_] := Flatten[Table[{-1.2,1.},{n/2}]];

Minimum[ExtendedRosenbrock, {n_, n_}] = {0};

Minimizer[ExtendedRosenbrock, {n_, n_}] := {Table[1, {n}]};

Condition[Residual[ExtendedPowell, {n_, n_} /; IntegerQ[n/4]][X_], VQ[X, n]] := Module[{x},Set[Evaluate[xl[x,n]], X];
Flatten[
    Table[{
            x[i4 - 3] + 10 x[i4 - 2], 
            Sqrt[5] (x[i4 - 1] - x[i4]), 
            (x[i4 - 2] - 2 x[i4 - 1])^2, 
            Sqrt[10] (x[i4 - 3] - x[i4])},
        {i4, 4, n, 4}]
]];

DefaultSize[ExtendedPowell] = {100, 100};

StartingPoint[ExtendedPowell, n_] := Flatten[Table[{3., -1., 0., 1.},{n/4}]];

Minimum[ExtendedPowell, {n_, n_}] = {0};

Minimizer[ExtendedPowell, {n_, n_}] := {Table[0,{n}]};


Condition[Residual[PenaltyFunctionI, {n_, m_} /; m == n + 1][X_], VQ[X, n]] := Module[{x},Set[Evaluate[xl[x,n]], X];
Append[
    Table[10^(-5/2) (x[i] - 1), {i, n}],
    Sum[x[j]^2, {j,n}] - 1/4]];
    
DefaultSize[PenaltyFunctionI] = {10, 11};

StartingPoint[PenaltyFunctionI, n_] := N[Range[n]];
 
Minimum[PenaltyFunctionI,{10, 11}] :=  {Root[-15234031801521 + 328764880316800016*#1 - 1607807998720000000000*#1^2 + 25600000000000000000000*#1^3 & , 1, 0]};
 
Minimizer[PenaltyFunctionI,{10, 11}] = {Table[Root[-1 - 49999*#1 + 2000000*#1^3 & , 3, 0],{10}]};

Condition[Residual[PenaltyFunctionII,{n_, m_} /; m == 2 n][X_], VQ[X, n]] := Module[{x},Set[Evaluate[xl[x,n]], X];
Join[
    {x[1] - 1/5},
    Table[10^(-5/2) (Exp[x[i]/10] + Exp[x[i-1]/10] - Exp[i/10] - Exp[(i - 1)/10]),
        {i, 2, n}],
    Table[10^(-5/2) (Exp[x[i - n + 1]/10] - Exp[-1/10]),
        {i, n + 1, 2 n - 1}],
    {Sum[(n - j + 1) x[j]^2,{j,1,n}] - 1}
]];

DefaultSize[PenaltyFunctionII] = {10,20};

StartingPoint[PenaltyFunctionII, n_] := Table[.5,{n}];

Minimum[PenaltyFunctionII, {10, 20}] = {0.0002936605374567459378822036216180360890033188423225600910654684252428267011920218312394844000141196469343890942097`100.};

Minimizer[PenaltyFunctionII, {10, 20}] = {
 {0.1999836051978236055003874012540132234899633728402244415438973157119886286388309195926362597974104070847472312124495`100., 
  0.0103506484712922978096838599856200355297606403048003356186362335456557003669405294544408600664235936437349672921293`100., 
  0.0196049344804384004971393614947231159960199815717401466890520435686081367701019905767769302335547217146526133088319`100., 
  0.0320890672206856500651689562808839307094639013690337024325893674012935191156988300089846923228363778906875492694622`100., 
  0.0499326773996412971694196426374632854782291432766069148049207289918620557192573105580145380827423198518120977413175`100., 
  0.076513995153993608158537011815129726336824835141545109126304760396358645477772977309516230002694511584261725194836`100., 
  0.1186240728695041803276967351109965021527976573718863261437050402180459365019117679031610337871217940882613226323705`100., 
  0.1921448723355766765131896365151244795323235400424497613745735002687627803042473710379599098658614525877221173790959`100., 
  0.3473205869418436335067486531998477006242190298548266257604446290395292572069087307247933099068545567085829999284637`100., 
  0.3691643741593508299261033862225334349242393245989018982379770759749790821548445773216751104893468863099312252723397`100.}};

Condition[Residual[VariablyDimensionedFunction, {n_, m_} /; m == n + 2][X_], VQ[X, n]] := Module[{tn, s},
    tn = X - 1;
    s = Tr[tn Range[n]];
    Join[tn,{s, s^2}]
];

DefaultSize[VariablyDimensionedFunction] = {100, 102};

StartingPoint[VariablyDimensionedFunction, n_] := 1. - Range[n]/n;
 
Minimum[VariablyDimensionedFunction, {n_, m_}] = {0};
 
Minimizer[VariablyDimensionedFunction, {n_, m_}] := {Table[1, {n}]};

Condition[Residual[TrigonometricFunction, {n_, n_}][X_], VQ[X, n]] := Module[{s, x},
    Set[Evaluate[xl[x,n]], X];
    s = Sum[x[j],{j,1,n}];
    Table[n - s + i (1 - Cos[x[i]]) - Sin[x[i]],{i,1, n}]
];

DefaultSize[TrigonometricFunction] = {10, 10};

StartingPoint[TrigonometricFunction, n_] := Table[1./n,{n}];

Minimum[TrigonometricFunction, {n_, n_}] = {0, 0}; 

Minimizer[TrigonometricFunction, {10, 10}] = {
 {-0.297614128722289479177580362628129283806840244343780618626158096933679772852893241990024113292174666381100567196295`100., 
  -0.2686709516449765024456895403919265267647493216310310810743941078977478117143433385498935807623712540487293584748078`100., 
  -0.2479715369801399757665641447231875270169858875187526119240504012669186330756828519868213377805596009198607515937801`100., 
  -0.2320585246370433023844804697368808950957164215264272340709786911808622204052718429950421911445500737403696684322894`100., 
  -0.219253785640394487963300834689537252899729446379821109968782431027410181323511668519725333474431133276309911704819`100., 
  -0.2086187141822497491357218678964049338738317075276738041157755150055000922600062764848727677734112361022930292887946`100., 
  -0.1995770681550349138943523405421764050945606953964594001072786786968312993289847355836703045456734597658837146129285`100., 
  -0.1917507705226575604727316212080473495783103957059943709923699582444045751794014720602401869575584178062319975089498`100., 
  -0.1848790481750464560371056715094290265559297325096216417443284183601796775152460806002397153169110772610278314011095`100., 
  12.3875957439261225735853186289846305551348325495626798567520309203661578004210488499689788320358324334110531390203984`100.},
 {2.5699312854337508236945572950962316991778670543881637833023243821667130497877608814172937360938219519618474908963718`100., 
  1.71602910374263420667473309466475974880218406789311295411741841165345864688974312939246257913485789068290392086486`100., 
  1.3249638177685855278495339945541423539347090730843220248625797351317528883612065350096977000442581645052569569639918`100., 
  1.1017873305983774328971514292950621523444116233719690840872169171936088275566266801738773090081101231420290644751575`100., 
  0.9562351683602417702503420259523760570173509369386283003037347460953103527117363658171191030010078500406988345280852`100., 
  0.8528933787134602186063147795259734933325496988008228639413512345534525230030476478365042842423592938194085264136177`100., 
  0.7751521475312680332835282608076795714099996977923703883612522823083687318770495819726897657239841082862887892901094`100., 
  0.7141835777222504384305093238184466955534571053612262312825497737109627742226169393749732598129639983960242764484881`100., 
  0.664849660039971097457125257890639743859985277442199063143680661359861329488929732044981506956443699154268871768026`100., 
   0.6239478054483922999252853087144027299525133805780082923737648880917157481715614649917836215847894299047058439505042`100.}};

Condition[Residual[BrownAlmostLinear, {n_, n_}][X_], VQ[X, n]] := Module[{s, x},
    Set[Evaluate[xl[x,n]], X];
    s = Sum[x[j], {j, 1, n}];
    Append[
        Table[x[i] + s - (n + 1),{i, 1, n - 1}],
        Product[x[j], {j, 1, n}] - 1]
];

DefaultSize[BrownAlmostLinear] = {10, 10};

StartingPoint[BrownAlmostLinear, n_] := Table[.5, {n}];

Minimum[BrownAlmostLinear, {n_, n_}] = {0, 1};

Minimizer[BrownAlmostLinear, {n_, n_}] := {Table[1,{n}], Append[Table[0,{n-1}], n + 1]};

Condition[Residual[DiscreteBoundaryValue, {n_, n_}][X_], VQ[X, n]] := Module[{h = 1/(n + 1), disc, x},
    Set[Evaluate[xl[x,n]], X];
    disc[i_] :=2 x[i] - x[i - 1] - x[i + 1] + h^2 (x[i] + i h + 1)^3/2;
    disc[1] = (disc[1] /. x[0]->0);
    disc[n] = (disc[n] /. x[n + 1]->0);
    Table[disc[i], {i, 1, n}]
];

DefaultSize[DiscreteBoundaryValue] = {10, 10};

StartingPoint[DiscreteBoundaryValue, n_] := Module[{h = 1/(n + 1)},
    Table[i h (i h - 1),{i, 1, n}]
];

Minimum[DiscreteBoundaryValue, {n_, n_}] := {0};

Minimizer[DiscreteBoundaryValue, {10, 10}] = { 
 {-0.0431649825187648705768864628079699058312963212456109956295815403923977557529105870945575739111982137908372346215894`100., 
  -0.0815771565353868815338983704308228851908380461910342675652803756917483485177777705692135700505778565319023477878897`100., 
  -0.1144857143805292872434083498306994794422402386107619723911886247617426405880016056477920701274057182874492379881582`100., 
  -0.1409735768625966796322772753046775485309952100535914493354448736970575810267835108267240487719711372653381381761247`100., 
  -0.1599086961819831223255723744073404986274397071783159010128437045037201254987296973901903752625158432062390303678591`100., 
  -0.169877202312774918976188661991685034378901099773706888229356214227411284670861143998868463825171194240184853088814`100., 
  -0.1690899837812083518441366540628699671441915021756790009776894789453503303154250282766294514793841406665180746093749`100., 
  -0.1552495352218318219469025370006293965974567742727304180343911067132848119071050917470448909508536230661490720765418`100., 
  -0.1253558916789349894004077915401957090898454148703984637559540807994684007330293483759898775170317300629943589567831`100., 
   -0.0754165336858920839547720879023882311927525115381388594976661543963005954848128258143122824278950138578595771564456`100.}};

Condition[Residual[DiscreteIntegralEquation, {n_, n_}][X_], VQ[X, n]] := Module[{h = 1/(n + 1), disc, x},
    Set[Evaluate[xl[x,n]], X];
    disc[i_] := x[i] + h ((1 - i h) Sum[j h (x[j] + j h + 1)^3, {j, 1, i}] + i h Sum[(1 - j h) (x[j] + j h + 1)^3, {j, i + 1, n}]);
    Table[disc[i], {i, 1, n}]
];

DefaultSize[DiscreteIntegralEquation] = {10,10};


StartingPoint[DiscreteIntegralEquation, n_] := Module[{h = 1/(n + 1)},
    Table[i h (i h - 1),{i, 1, n}]
];

Minimum[DiscreteIntegralEquation, {n_, n_}] = {0};

Minimizer[DiscreteIntegralEquation, {10, 10}] = {
 {-0.0715682258408752130395722243779731848796843113932050971950245956214572256006168214098583049564621427593062873407551`100., 
  -0.1343831290601913003487072812267835666772928074827646316706240611334017208117460982508174458797728741121901107253412`100., 
  -0.1877008244792710336798262205802779562470691458220681747877686746888308238707747935553477804393494795101184913819824`100., 
  -0.2304616394529525492108806531924680098985362369334276575479682001727658951425752392665211814672007889457022283951337`100., 
  -0.2611968956224301074473006687333478349460013530608943358946922720402797875231336242711039654520161014454509920697306`100., 
  -0.2778873191449419071755571679377604148526419702803715176090178368840276114323731445185704891834194180044056153369554`100., 
  -0.2777460561170090145402749532148983111249812173479958604246298685970414610589839231354215783186575119646835956105501`100., 
  -0.2568792309084578321052486261405581820277515496007401628391129329110805485899792167678703354052580916811419984109561`100., 
  -0.2097390618835158862278245309627414889037297647859569775253007837467419792504127658684957421431047433600305890617943`100., 
   -0.1282089497609331033693501133726443613438387402582804513418199537426519080714663806189647743606044793156520508143248`100.}};

Condition[Residual[BroydenTridiagonal, {n_, n_}][X_], VQ[X, n]] := Module[{disc, x},
    Set[Evaluate[xl[x,n]], X];
    disc[i_] := (3 - 2 x[i]) x[i] - x[i - 1] - 2 x[i + 1] + 1;
    disc[1] = (disc[1] /. x[0] ->0);
    disc[n] = (disc[n] /. x[n + 1]->0);
    Array[disc, n]
];

DefaultSize[BroydenTridiagonal] = {10, 10};

StartingPoint[BroydenTridiagonal, n_] := Table[-1,{n}];
 
Minimum[BroydenTridiagonal, {n_, n_}] = {0};

Minimizer[BroydenTridiagonal, {n_, n_}] =  {
 {-0.5707221320112247936619691310953214600106448708881003252026269047333463121744653803783448775309063665062120561896761`100., 
  -0.6818069499842750908331179122695546901572960969856426119143566324686033162421644259828095347081211309399409327238725`100., 
  -0.7022100760176600347027417411347233198537360235585628127120094899572262773627033566623374630488584040849769058692246`100., 
  -0.7055106298950803912594153632941824546260473303704974675603008693523023554194270835853571195863226994117676220856413`100., 
  -0.704906155728743671024913551450904463905000297180482809848896090374880992027560443348694614151749710134189818112644`100., 
  -0.7014966070298511346842434251094363692762765457429091256269132374980069772963958950595507332569844815251172790230937`100., 
  -0.6918893223547982549069930103332955962326393191417564610435167874868359239738530561111356155430828436682474543242293`100., 
  -0.6657965144058537472130147385867585676523446434361555968840013364974730841641509509863215574913697944841463122373065`100., 
  -0.5960351090263657097072677236037060038560239380233851439457682792942390104569162108007397865529109309417073707928759`100., 
   -0.4164122575286933492735567182211811950539060022197650159937818194191798863560981367725828599277451561468399057344964`100.}};

Condition[Residual[BroydenBanded, {n_, n_}][X_], VQ[X, n]] := Module[{x},Set[Evaluate[xl[x,n]], X];
    Table[x[i] (2 + 5 x[i]^2) + 1 - (
            Sum[x[j] (1 - x[j]), {j, Max[1, i - 5], i - 1}] + 
            Sum[x[j] (1 - x[j]), {j, i + 1, Min[n, i + 1]}]),
        {i, 1, n}]];

DefaultSize[BroydenBanded] = {10, 10};

StartingPoint[BroydenBanded, n_] := Table[-1,{n}];

Minimum[BroydenBanded, {n_, n_}] = {0};

Minimizer[BroydenBanded, {n_, n_}] = {
 {-0.606365679915713397776037488696357555566874355270083452928746213719019066872139430887378450699592044665237515397795`100., 
  -0.7559745292202371861595571756573360857821890312159830485060712609272428866859357059681499854081057397562195673118006`100., 
  -0.8957405797956297204054944790893664673604233143554508942467258563398245206208616295523598228107112774639660210188497`100., 
  -1.0275656149223761562374530974572465030557237135783973887695688913377065922059551281962780412150407151381728123161812`100., 
  -1.1524252870278769677890657499498299067470046656710476332174662787551482994022283700649951385454793137497951300106294`100., 
  -1.2675896672781640860903844757664158149481879077115782874341563005190199409945002754417440378552151717453170979739537`100., 
  -1.3453497543165145870110290496966652631698025056952154263808001315840339150199925090754023050299380777675587385599136`100., 
  -1.4110161598593014417740707976501863615147672315696854602648408057223318610371135106246806634193852151084332006133069`100., 
  -1.4561993314790667549095014240368296533274766989080719444093000004381756013933694888933300027965683564912260919788519`100., 
   -1.3990087973659092552964740858781748487431194958058758243482163080113129629702073791152418545555030648992224291634122`100.}};

Condition[Residual[LinearFullRank, {n_, m_} /; m >= n][X_], VQ[X, n]] :=  Module[{s, x},
    Set[Evaluate[xl[x,n]], X];
    s = 2 Sum[x[j], {j, 1, n}]/m + 1;
    Join[
        Table[x[i] - s, {i, 1, n}],
        Table[-s, {i, n + 1, m}]]
];

DefaultSize[LinearFullRank] = {10, 20};

StartingPoint[LinearFullRank, n_] := Table[1., {n}];

Minimum[LinearFullRank, {n_, m_}] := {m - n};

Minimizer[LinearFullRank, {n_, m_}] := {Table[-1,{n}]};


Condition[Residual[LinearRank1, {n_, m_} /; m >= n][X_], VQ[X, n]] :=  Module[{s, x},
    Set[Evaluate[xl[x,n]], X];
    s = Sum[j x[j], {j, 1, n}];
    Table[i s - 1, {i, 1, m}]
];

DefaultSize[LinearRank1] = {10, 20};

StartingPoint[LinearRank1, n_] := Table[1., {n}];

Minimum[LinearRank1, {n_, m_}] := {m (m - 1)/(2 (2 m + 1))};

Minimizer[LinearRank1, {n_, m_}] := {x_List /; Apply[Plus, Range[n] x] == 3/(2 m + 1)};

Condition[Residual[LinearRank1Z, {n_, m_} /; m >= n][X_], VQ[X, n]] :=  Module[{s, x},
    Set[Evaluate[xl[x,n]], X];    
    s = Sum[j x[j], {j, 2, n - 1}];
    Join[{-1}, Table[(i - 1) s - 1, {i, 2, m - 1}], {-1}]
];

DefaultSize[LinearRank1Z] = {10, 20};

StartingPoint[LinearRank1Z, n_] := Table[1., {n}];

Minimum[LinearRank1Z, {n_, m_}] := {(m^2 + 3 m - 6)/(2 (2 m - 3))};

Minimizer[LinearRank1Z, {n_, m_}] := {x_List /; Apply[Plus, Range[n] x] == 3/(2 m - 3)};

Condition[Residual[Chebyquad, {n_, m_} /; m >= n][X_], VQ[X, n]] := Module[{T, TI, x},
    Set[Evaluate[xl[x,n]], X];    
    T[i_, x_] := ChebyshevT[i, 2 x - 1];
    TI[i_?OddQ] = 0;
    TI[i_?EvenQ] := -1/(i^2 - 1);
    Table[Sum[T[i, x[j]], {j, 1, n}]/n - TI[i],{i, 1, n}]
];

DefaultSize[Chebyquad] = {9, 9};

StartingPoint[Chebyquad, n_] := N[Range[n]/(n + 1)];

Minimum[Chebyquad, {9, 9}] = {0};

Minimizer[Chebyquad, {9, 9}] = {
 {0.0442053461357827631675257160839500895241775966492397447457326023078969435877752364532102229441744612102222410290034`100., 
  0.1994906723098809642859360339325006269067574611216582073625665368642322118236405219559393891123700432357954735115214`100., 
  0.2356191084710600033699091893227915045169958433629658699725764190991644616394650692642455591790267080768754749175746`100., 
  0.4160469078925980284659840508725689920632769385300696595154723159786274462095181585938322401564839245510933309795943`100., 
  0.5`100., 
  0.5839530921074019715340159491274310079367230614699303404845276840213725537904818414061677598435160754489066690204057`100., 
  0.7643808915289399966300908106772084954830041566370341300274235809008355383605349307357544408209732919231245250824254`100., 
  0.8005093276901190357140639660674993730932425388783417926374334631357677881763594780440606108876299567642045264884786`100., 
  0.9557946538642172368324742839160499104758224033507602552542673976921030564122247635467897770558255387897777589709966`100.}};


$UnconstrainedProblems = Cases[SubValues[Residual], Residual[name_Symbol, size_][_] :> ((ProblemSizePattern[name] = size); name), Infinity];

$FindMinimumProblems = $UnconstrainedProblems;

$FindRootProblems = {Rosenbrock, FreudensteinRoth, PowellSingular, PowellBadlyScaled, HelicalValley, Watson, Chebyquad, BrownAlmostLinear, DiscreteBoundaryValue, DiscreteIntegralEquation, TrigonometricFunction, BroydenTridiagonal, BroydenBanded, Gulf, Box3D, BiggsExp6, ExtendedRosenbrock, ExtendedPowell};

checkContext[ sym_] :=
    If[ Context[ sym] =!= "Optimization`UnconstrainedProblems`",
        Message[ UnconstrainedProblem::context, sym]]

Map[ checkContext, $UnconstrainedProblems];



SetAttributes[AverageTiming, HoldAll];
AverageTiming[expr_] := 
  Module[{tries = 1, t}, Times[Block[{Second = 1}, t = First[Timing[expr]];
        While[t < 1, tries *= 2; t = First[Timing[Do[expr, {tries}]]]];
        t/tries] , Second]];



ToDigits[err_] := Module[{prec = Precision[err]},
    If[prec === MachinePrecision,
        If[err == 0., 
            N[MachinePrecision],
            -Log[10., err]
        ],
    (* else *)
        If[Developer`ZeroQ[error],
            Accuracy[error],
            -N[Log[10, err]]
        ]
    ]
];


APRules[err_, exact_, name_] := Module[{prule},
    If[Developer`ZeroQ[exact],
        prule = ("FunctionPrecision" -> Indeterminate),
        prule = ("FunctionPrecision" -> ToDigits[err/exact])];
    {"FunctionAccuracy"->ToDigits[err], prule}
]
    

CheckSolution[ prob:FindMinimumProblem[pfun_, pvars_List, opts_, problem_, size_, ___], {fsol_, xsolrules_, ___}] := 
    Module[{minima, errs, order, err, prec, xsol, xacc, xprec},
        prec = Internal`EffectivePrecision[{fsol, xsolrules}];
        xsol = xsolrules[[All, 2]];
        If[(Length[xsol] == 1) && (Length[xsol[[1]]] == First[size]),
            xsol = First[xsol]];
        minima = Minimum[problem, size];
        minimizers = Minimizer[problem, size];
        If[Or[Head[minima] === Minimum, Head[minimizers] === Minimizer],
            Message[ProblemTest::unksz, problem];
            sol = ProblemSolve[prob,  Method->Newton, WorkingPrecision->3*prec, 
                    PrecisionGoal->1.5*prec, AccuracyGoal->1.5*prec];
            minima = {sol[[1]]};
            minimizers = {sol[[2, All, 2]]}
        ];
        errs = Map[Abs[fsol - #]&, minima];
        order = First[Ordering[errs, 1]];
        err = errs[[order]];
        frules = APRules[err, minima[[order]], "f"];
        minimizers = minimizers[[order]];
        If[VectorQ[minimizers, NumericQ],
            err = xsol - minimizers;
            xacc = ToDigits[Norm[err, 2]];
            xprec = ToDigits[Norm[err/Internal`MaxAbs[minimizers, SetPrecision[10^-prec, prec]],2]];
            err = Norm[xsol - minimizers, 2];
            Join[frules, {"SpatialAccuracy"->xacc, "SpatialPrecision"->xprec}],
            Join[frules, {"SpatialAccuracy"->"ERROR", "SpatialPrecision"->"ERROR"}]
        ]
    ]

CheckSolution[ prob:FindRootProblem[pfun_, pvars_List, opts_, problem_, size_, ___], xsolrules_] := 
    Module[{minima, errs, order, err, prec, xsol, xacc, xprec},
        prec = Internal`EffectivePrecision[{xsolrules}];
        err = Norm[First[pfun] /. xsolrules, 2];
        frules = {"FunctionAccuracy"->ToDigits[err]};
        minima = Minimum[problem, size];
        roots = Minimizer[problem, size];
        xsol = xsolrules[[All, 2]];
        If[Or[Head[minima] === Minimum, Head[roots] === Minimizer],
            Message[ProblemTest::unksz, problem];
            sol = ProblemSolve[prob, Method->Newton, WorkingPrecision->3*prec, 
                    PrecisionGoal->1.5*prec, AccuracyGoal->1.5*prec];;
            minima = {0};
            roots = {sol[[All, 2]]}];
        order = Position[minima, 0][[1,1]];
        roots = roots[[order]];
        If[VectorQ[roots, NumericQ],
            err = xsol - roots;
            xacc = ToDigits[Norm[err, 2]];
            xprec = ToDigits[Norm[err/Internal`MaxAbs[roots, SetPrecision[10^-prec, prec]], 2]];
            err = Norm[xsol - roots, 2];
            Join[frules, {"SpatialAccuracy"->xacc, "SpatialPrecision"->xprec}],
            Join[frules, {MatchQ[xsol, roots]}]
        ]
    ]


(*
 Implementation of ProblemSolve, ProblemTime, ProblemTest, ProblemStatistics
*)

FindWhat[FindMinimumProblem] = FindMinimum;
FindWhat[FindRootProblem] = FindRoot;

(prob:FindMinimumProblem[args___])["Solve"[opts___]] := ProblemSolve[prob, opts];

(prob:FindRootProblem[args___])["Solve"[opts___]] := ProblemSolve[prob, opts];

ProblemSolve[ findp_[pfun_, pvars_,{popts___}, ___], opts___] := FindWhat[findp][pfun, pvars, opts, popts]

(prob:FindMinimumProblem[args___])["Time"[opts___]] := ProblemTime[prob, opts];

(prob:FindRootProblem[args___])["Time"[opts___]] := ProblemTime[prob, opts];

ProblemTime[ findp_[pfun_, pvars_, {popts___}, ___], opts___] := 
    Module[ {t, sol},
        Block[{$Messages = {}},
            t = AverageTiming[sol = FindWhat[findp][pfun, pvars, opts, popts]]];
        {sol, "Time"->t}
    ]

(prob:FindMinimumProblem[args___])["Statistics"[opts___]] := ProblemStatistics[prob, opts];

(prob:FindRootProblem[args___])["Statistics"[opts___]] := ProblemStatistics[prob, opts];

ProblemStatistics[prob:(findp_[pfun_, pvars_, {popts___}, ___]), opts___] := 
Module[{flops, stats, f = 0, s = 0, g = 0, h = 0, r = 0, j = 0, reset},
    flops = GetOptions[FindWhat[findp], {opts, popts}];
    flops = PrependToMonitor[s++, "StepMonitor", flops];
    flops = PrependToMonitor[f++, "EvaluationMonitor", flops];
    If[findp === FindMinimumProblem, 
        flops = AddToDerivativeOption[g++, "Gradient", flops];
        flops = HandleLMOpts[{r++, j++}, flops, reset];
        flops = HandleHessian[{h++}, flops],
    (* else *)
        flops = AddToDerivativeOption[j++, "Jacobian", flops]
    ];
    res = CheckAbort[
            ProblemSolve[prob, flops],
                ResetLMOpts[reset]; $Aborted];
    ResetLMOpts[reset];
    stats = DeleteCases[{"Steps" -> s, "Function" -> f, "Gradient" -> g, "Hessian" -> h, "Residual"->r, "Jacobian"->j}, Rule[_String, 0]];
    Append[res, stats]
    ]

(prob:FindMinimumProblem[args___])["Test"[opts___]] := ProblemTest[prob, opts];

(prob:FindRootProblem[args___])["Test"[opts___]] := ProblemTest[prob, opts];

ProblemTest[prob:(findp_[pfun_, pvars_, {popts___}, ___]), opts___] := 
    Module[{n, m, sol, obj, args, t, mess},
        Block[{$MessageList = {}},
            count = Last[ProblemStatistics[prob, opts]];
            mess = $MessageList];
        time = ProblemTime[ prob, opts];
        sol = First[time];
        time = Take[time, -1];    
        err = CheckSolution[ prob, sol];
        Flatten[{err, time, count, Messages->mess}]
    ]
       

(*
    Basic implementation of GetFindMinimumProblem. 
    The actual work is done by GetFindMinimumProblemImpl, which 
    issues messages and throws exceptions if some illegal input 
    is given.


*)

Clear[ GetFindMinimumProblem, GetFindMinimumProblemImpl]

Options[GetFindMinimumProblem] = {Variables->Function[Subscript[X, #]]};

GetFindMinimumProblem[ args___] := 
    Module[ {ef},
        ef = 
            Catch[
                GetProblem[FindMinimumProblem, Options[GetFindMinimumProblem], args], 
                    _UnconstrainedProblemException, $Failed&];
        ef /; ef =!= $Failed
        ]

Options[GetFindRootProblem] = {Variables->Function[Subscript[X, #]]};

GetFindRootProblem[ args___] := 
    Module[ {ef},
        ef = 
            Catch[
                GetProblem[FindRootProblem, Options[GetFindRootProblem], args], 
                    _UnconstrainedProblemException, $Failed&];
        ef /; ef =!= $Failed
        ]

GetProblem[findp_, gpopts_, problem_, args___] := 
    Module[{res},
        CheckKnown[findp, problem];
        res = GetProblemImpl[findp, problem, args, Apply[Sequence, gpopts]];
        If[Head[res] === GetProblemImpl, 
            MessageToss[findp, "args", {args}]];
        res
    ];

GetProblem[findp_, gpopts_] := MessageToss[findp, "argb", findp, 0, 1, 4];

MessageToss[findp_, mname_, margs___] := 
    CompoundExpression[
        Message[MessageName[findp, mname], margs],
        Throw[1, UnconstrainedProblemException[]]
    ]

CheckKnown[FindMinimumProblem, problem_] := 
    If[Not[MemberQ[$FindMinimumProblems, problem]], 
        MessageToss[FindMinimumProblem, "unk", problem]];

CheckKnown[FindRootProblem, problem_] := 
    If[Not[MemberQ[$FindRootProblems, problem]], 
        MessageToss[FindRootProblem, "unk", problem]];

CheckSize[FindMinimumProblem, problem_, size_] := 
    If[Not[MatchQ[size, ProblemSizePattern[problem]]],
        MessageToss[FindMinimumProblem, "size", size, problem, ProblemSizePattern[problem]],
        size
    ];

CheckSize[FindRootProblem, problem_, size1_] := 
    If[Not[MatchQ[{size1, size1}, ProblemSizePattern[problem]]],
        MessageToss[FindRootProblem, "size", size1, problem, ProblemSizePattern[problem]],
        {size1, size1}
    ];

getStartingPoint[findp_, problem_, size_] := 
    StartingPoint[problem, First[CheckSize[findp, problem, size]]];

GetProblemImpl[FindMinimumProblem, problem_, opts___?OptionQ] := 
    GetProblemImpl[FindMinimumProblem, problem, DefaultSize[problem], opts];

(*
    Special case because m is fixed for Watson
*)
GetProblemImpl[FindRootProblem, Watson, opts___?OptionQ] := 
    GetProblemImpl[FindRootProblem, Watson, Last[DefaultSize[Watson]], opts];

GetProblemImpl[FindRootProblem, problem_, opts___?OptionQ] := 
    GetProblemImpl[FindRootProblem, problem, First[DefaultSize[problem]], opts];

GetProblemImpl[findp_, problem_, size_, opts___?OptionQ] := 
    GetProblemImpl[findp, problem, size, getStartingPoint[findp, problem, size], opts];

GetProblemImpl[findp_, problem_, size_, start_?NumberQ, opts___?OptionQ] := 
    GetProblemImpl[findp, problem, size, start*getStartingPoint[findp, problem,size], opts];


GetProblemImpl[findp_, problem_, size_, start_, opts___?OptionQ] :=
    Module[{n, m, vars, vfun, fun, vin},
        {n, m} = CheckSize[findp, problem, size];
        vfun = Variables /. Flatten[{opts, Options[GetFindMinimumProblem]}];
        If[ !ListQ[ start] || Length[start] != n,
            MessageToss[findp, "start", start, problem, {n, m}]];
        If[ListQ[vfun],
            If[Length[vfun] < n, MessageToss[findp, "vars", vfun, n]];
            vars = Take[vfun, n],
            vars = Array[vfun, n]
        ];
        vin = Map[Flatten, Transpose[{vars, start}]];
        fun = If[findp === FindMinimumProblem, Objective, Residual];
        findp[
            fun[problem, {n, m}][vars], 
            vin,                  
            {FilterOptions[FindMinimum, opts]},
            problem, 
            {n, m}]
    ]


GetProblemImpl[findp_,  args___] := MessageToss[findp, "args", {args}];

(* 

 Version of FilterOptions that works for strings.

*)

GetOptions[sym_, opts___] := 
    ProcessOptionNames[Flatten[{FilterAllOptions[sym, opts], Options[sym]}]];


(* Convert symbol optionnames into strings. *)

SetAttributes[ProcessOptionNames, Listable]; 

ProcessOptionNames[(r : (Rule | RuleDelayed))[name_Symbol, val_]] := 
    r[SymbolName[name], val];
    
ProcessOptionNames[opt_] := opt;

StringName[sym_Symbol] := SymbolName[sym];
StringName[name_] := name;

FilterAllOptions[ command_Symbol, options___ ] :=
	FilterAllOptions[ ProcessOptionNames[ Options[command]][[All,1]], options ]

FilterAllOptions[ opts_List, options___ ] :=
	Select[ ProcessOptionNames[Flatten[{options}]], MemberQ[opts, First[#]]& ]



(*

 Special processing needed for LevenberqMarquardt
 
*)

SetAttributes[HandleLMOpts, HoldAll];
HandleLMOpts[{sowr_, sowj_}, flops_, reset_] := 
    Module[{LMopts, mopts = {}, method},
        method = "Method" /. flops;
        If[ListQ[method], 
            mopts = Rest[method];
            method = First[method];
        ];
        method = StringName[method];
        If[Not[Or[SameQ[method, "Automatic"], SameQ[method, "LevenbergMarquardt"]]],
            Return[flops];
        ];
        If[SameQ[method, "Automatic"],
            (* We have to reset Options[LevenberqMarquardt] *)
            reset = Options[FindMinimum`LevenbergMarquardt];
            LMopts = reset,
            LMopts = Flatten[{mopts, Options[FindMinimum`LevenbergMarquardt]}]
        ];
        LMopts = AddToDerivativeOption[sowj, "Jacobian", ProcessOptionNames[LMopts]];
        LMopts = PrependToMonitor[sowr, "EvaluationMonitor", LMopts];
        If[ListQ[reset],
            SetOptions[FindMinimum`LevenbergMarquardt, Apply[Sequence, Flatten[Reverse[LMopts]]]],
        (* else *)
            flops = Prepend[flops, Method->Prepend[LMopts, "LevenbergMarquardt"]];
        ];
        flops
    ];

ResetLMOpts[reset_] := If[ListQ[reset], SetOptions[FindMinimum`LevenbergMarquardt, Apply[Sequence, reset]]];

SetAttributes[HandleHessian, HoldAll];
HandleHessian[{sowh_}, flops_] := 
    Module[{Newtonopts, mopts = {}, method},
        method = "Method" /. flops;
        If[ListQ[method], 
            mopts = Rest[method];
            method = First[method];
        ];
        method = StringName[method];
        If[Not[SameQ[method, "Newton"]],
            Return[flops];
        ];
        Newtonopts = Flatten[{mopts, Options[Newton]}];
        Newtonopts = AddToDerivativeOption[sowh, "Hessian", ProcessOptionNames[Newtonopts]]; 
        flops = Prepend[flops, Method->Prepend[Newtonopts, "Newton"]]
    ];


(*
 Prepend a new function to an existing monitor option (evaluation or step monitor)
*)

SetAttributes[PrependToMonitor, HoldFirst];

PrependToMonitor[dofirst_, monitor_, flops_] := 
    Module[{pos, res},
        pos = Position[flops, (Rule | RuleDelayed)[monitor, _], {1}];
        If[pos === {}, 
            Join[flops, {RuleDelayed[monitor, dofirst]}],
            pos = pos[[1,1]];
            res = flops;
            res[[pos]] = MonitorPrepend[dofirst, flops[[pos]]];
            res
        ]
    ];

SetAttributes[MonitorPrepend, HoldFirst];

MonitorPrepend[dofirst_, (h:(Rule | RuleDelayed))[monitor_, None]] := 
    RuleDelayed[monitor, dofirst]

MonitorPrepend[dofirst_, (h:(Rule | RuleDelayed))[monitor_, stuff_]] := 
    RuleDelayed[monitor, CompoundExpression[dofirst, stuff]]


(*
 Add to the monitor of the derivative computation.
*)

SetAttributes[AddToDerivativeOption, HoldFirst];

AddToDerivativeOption[thing_, der_, flops_] := 
    Module[{dmethod, dopts = {}},
        dmethod = der /. flops;
        If[SameQ[dmethod, der], 
            dspec = der->{"Symbolic", EvaluationMonitor:>thing},
    (* else *)
            If[MatchQ[dmethod, {_,__?OptionQ} | {_String} | {_Symbol}],
                dopts = Rest[dmethod];
                dmethod = First[dmethod]
            ];
            dopts = PrependToMonitor[thing, "EvaluationMonitor", ProcessOptionNames[dopts]];
            dspec = der->Prepend[dopts, dmethod];
        ];
        Prepend[flops, dspec]
    ]
    
    
(*
 FindMinimumPlot
*)

(*
    Colors and sizes for evaluation point types
*)
sstyle = Apply[Sequence, {RGBColor[0,0,1], PointSize[0.015]}];
fstyle = Apply[Sequence, {RGBColor[0,1,0], PointSize[0.02]}];
gstyle = Apply[Sequence, {RGBColor[1,0,0], PointSize[0.0325]}];
hstyle = Apply[Sequence, {RGBColor[0, 1, 1], PointSize[0.045]}];
rstyle = Apply[Sequence, {RGBColor[1, 1, 0], PointSize[0.025]}];
jstyle = Apply[Sequence, {RGBColor[.5, 0, .5], PointSize[0.04]}];
endstyle = Apply[Sequence, {RGBColor[0,0,0], PointSize[0.055]}];

(prob:FindMinimumProblem[args___])["Plot"[opts___]] := FindMinimumPlot[prob, opts];

FindMinimumPlot[ FindMinimumProblem[ fun_, start_, {popts___},___], opts___] :=
    FindMinimumPlot[ fun, start, opts, popts]

FindMinimumPlot[obj_, {{x_, xst__}, {y_, yst__}}, opts___] :=
Module[{flops, f, s, g, h, r, j, all, res, minpt, allx, ally, xran, yran, reset, stats},
    flops = GetOptions[FindMinimum, opts];
    flops = PrependToMonitor[Sow[{x,y},{s, all}], "StepMonitor", flops];
    flops = PrependToMonitor[Sow[{x,y},{f, all}], "EvaluationMonitor", flops];
    flops = AddToDerivativeOption[Sow[{x,y}, {g, all}], "Gradient", flops];
    flops = HandleLMOpts[{Sow[{x,y}, {r, all}], Sow[{x,y}, {j, all}]}, flops, reset];
    flops = HandleHessian[{Sow[{x,y}, {h, all}]}, flops];
    {res, rules} = Reap[
        Sow[{First[{xst}], First[{yst}]},{s, all}];
        res = CheckAbort[
            FindMinimum[obj, {x, xst}, {y, yst}, 
                Evaluate[flops]],
            ResetLMOpts[reset]; $Aborted],
         _, 
         Rule];
    ResetLMOpts[reset];
    If[Not[ListQ[res]], Return[res]];
    minpt = {x,y} /. Last[res];
    {f, s, g, h, r, j, all} = {f, s, g, h, r, j, all} /. rules;
    stats = {"Steps" -> Length[s] - 1, "Function" -> Length[f], "Gradient" -> Length[g], "Hessian" -> Length[h], "Residual"->Length[r], "Jacobian"->Length[j]};
    stats = DeleteCases[stats, Rule[_String, 0]];
    {allx, ally} = Transpose[all];
    xran = MakeRange[x, allx];
    yran = MakeRange[y, ally];
    f = If[ListQ[f], {fstyle, Map[Point, f]}, {}];
    r = If[ListQ[r], {rstyle, Map[Point, r]}, {}];
    s = Prepend[s, {First[{xst}], First[{yst}]}];
    s = {sstyle, Line[s], Map[Point, s]};
    g = If[ListQ[g], {gstyle, Map[Point, g]}, {}];
    h = If[ListQ[h], {hstyle, Map[Point, h]}, {}];
    j = If[ListQ[j], {jstyle, Map[Point, j]}, {}];
    minpt = {endstyle, Point[minpt]};
    {res, stats, ContourPlot[obj, Evaluate[xran], Evaluate[yran], 
        Epilog -> {minpt, h, g, f, j, r, s}, 
        Evaluate[FilterOptions[ContourPlot,  opts]],
        ColorFunction->Function[GrayLevel[1 - .75 #]],
        ColorFunctionScaling->True,
        PlotPoints->100]}
    ]

FindMinimumPlot[obj_, {x_, xst__}, opts___] := FindMinimumPlot[obj, {{x, xst}}, opts];

FindMinimumPlot[obj_, {{x_, xst__}}, opts___] :=
Module[{flops, f, s, g, h, r, j, all, res, minpt, xran, reset},
    flops = GetOptions[FindMinimum, opts];
    flops = PrependToMonitor[Sow[{x, obj},{s, all}], "StepMonitor", flops];
    flops = PrependToMonitor[Sow[{x, obj},{f, all}], "EvaluationMonitor", flops];
    flops = AddToDerivativeOption[Sow[{x, obj}, {g, all}], "Gradient", flops];
    flops = HandleLMOpts[{Sow[{x, obj}, {r, all}], Sow[{x, obj}, {j, all}]}, flops, reset];
    flops = HandleHessian[{Sow[{x,obj}, {h, all}]}, flops];
    {res, rules} = Reap[
        Sow[{x, obj} /. x->First[{xst}],{s, all}];
        res = CheckAbort[
            FindMinimum[obj, {x, xst},Evaluate[flops]],
            ResetLMOpts[reset]; $Aborted],
        _, 
        Rule];
    ResetLMOpts[reset];
    If[Head[res] === FindMinimum, Return[res]];
    minpt = Reverse[res];
    minpt[[1]] = x /. res[[2]];
    {f, s, g, h, r, j, all} = {f, s, g, h, r, j, all} /. rules;
    stats = {"Steps" -> Length[s] - 1, "Function" -> Length[f], "Gradient" -> Length[g], "Hessian" -> Length[h], "Residual"->Length[r], "Jacobian"->Length[j]};
    stats = DeleteCases[stats, Rule[_String, 0]];
    all = all[[All, 1]];
    xran = MakeRange[x, all];
    If[TrueQ[Apply[Equal, Drop[xran, 1]]], xran[[{2,3}]] = {xran[[2]] - 1, xran[[2]] + 1}];
    f = If[ListQ[f], {fstyle, Map[Point, f]}, {}];
    r = If[ListQ[r], {rstyle, Map[Point, r]}, {}];
    s = Prepend[s, {x, obj} /. x -> First[{xst}]];
    s = {sstyle, Line[s], Map[Point, s]};
    g = If[ListQ[g], {gstyle, Map[Point, g]}, {}];
    h = If[ListQ[h], {hstyle, Map[Point, h]}, {}];
    j = If[ListQ[j], {jstyle, Map[Point, j]}, {}];
    minpt = {endstyle, Point[minpt]};
    {res, stats, Plot[obj, Evaluate[xran], Epilog -> {minpt, h, j, g, r, f, s}, 
        Evaluate[FilterOptions[Plot,  opts]]]}
    ]


(*
    FindRootPlot -- the code is just enough different from FindMinimumPlot
                    to make it worth having is separate.
*)
(prob:FindRootProblem[args___])["Plot"[opts___]] := FindRootPlot[prob, opts];

FindRootPlot[ FindRootProblem[ fun_, start_, {popts___},___], opts___] :=
    FindRootPlot[ fun, start, opts, popts]

FindRootPlot[obj_, {{x_, xst__}, {y_, yst__}}, opts___] :=
Module[{flops, s, r, j, all, res, minpt, allx, ally, xran, yran},
    flops = GetOptions[FindRoot, opts];
    flops = PrependToMonitor[Sow[{x,y},{s, all}], "StepMonitor", flops];
    flops = PrependToMonitor[Sow[{x,y},{r, all}], "EvaluationMonitor", flops];
    flops = AddToDerivativeOption[Sow[{x,y}, {j, all}], "Jacobian", flops];
    {res, rules} = Reap[
        Sow[{First[{xst}], First[{yst}]},{s, all}];
        res = FindRoot[obj, {x, xst}, {y, yst}, Evaluate[flops]],
         _, 
        Rule];
    If[Head[res] === FindRoot, Return[res]];
    root = {x, y} /. res;
    {s, r, j, all} = {s, r, j, all} /. rules;
    stats = {"Steps" -> Length[s], "Residual" -> Length[r],  "Jacobian" -> Length[j]};
    stats = DeleteCases[stats, Rule[_String, 0]];
    {allx, ally} = Transpose[all];
    xran = MakeRange[x, allx];
    yran = MakeRange[y, ally];
    s = {sstyle, Line[s], Map[Point, s]};
    r = If[ListQ[r], {rstyle, Map[Point, r]}, {}];
    j = If[ListQ[j], {jstyle, Map[Point, j]}, {}];
    root = {endstyle, Point[root]};
    {res, stats, 
      Show[Block[{$DisplayFunction = Identity},
          {ContourPlot[obj.obj/2, Evaluate[xran], Evaluate[yran], 
                ColorFunction->Function[GrayLevel[1 - .75 #]],
                ColorFunctionScaling->True,
                PlotPoints->100],
            ContourPlot[First[obj], Evaluate[xran], Evaluate[yran], 
              Contours -> {0}, ContourShading -> False, PlotPoints -> 100, 
              ContourStyle -> RGBColor[1, 0, 0]], 
            ContourPlot[Last[obj], Evaluate[xran], Evaluate[yran], 
              Contours -> {0}, ContourShading -> False, PlotPoints -> 100, 
              ContourStyle -> RGBColor[0, 1, 0]]}], 
            Epilog -> {root, j, r, s}, 
            Evaluate[FilterOptions[Show,  opts]]]}
    ]

FindRootPlot[inobj_, {x_, xst__}, opts___] := FindRootPlot[inobj, {{x, xst}}, opts];

FindRootPlot[inobj_, {{x_, xst__}}, opts___] :=
Module[{flops, s, r, j, all, res, minpt, xran},
    obj = If[ListQ[inobj], First[inobj], inobj];
    flops = GetOptions[FindRoot, opts];
    flops = PrependToMonitor[Sow[{x, obj},{s, all}], "StepMonitor", flops];
    flops = PrependToMonitor[Sow[{x, obj},{r, all}], "EvaluationMonitor", flops];
    flops = AddToDerivativeOption[Sow[{x, obj}, {j, all}], "Jacobian", flops];
    {res, rules} = Reap[
        Sow[{x, obj} /. x->First[{xst}],{s, all}];
        res = FindRoot[obj, {x, xst}, Evaluate[flops]],
        _, 
        Rule];
    If[Head[res] === FindRoot, Return[res]];
    root = {x, obj} /. res;
    {s, r, j, all} = {s, r, j, all} /. rules;
    stats = {"Steps" -> Length[s], "Residual" -> Length[r],  "Jacobian" -> Length[j]};
    stats = DeleteCases[stats, Rule[_String, 0]];
    all = all[[All, 1]];
    xran = MakeRange[x, all];
    If[TrueQ[Apply[Equal, Drop[xran, 1]]], xran[[{2,3}]] = {xran[[2]] - 1, xran[[2]] + 1}];
    s = {sstyle, Line[s], Map[Point, s]};
    r = If[ListQ[r], {rstyle, Map[Point, r]}, {}];
    j = If[ListQ[j], {jstyle, Map[Point, j]}, {}];
    root = {endstyle, Point[root]};
    {res, stats, 
      Plot[obj, Evaluate[xran], Epilog -> {root, j, r, s}, 
        Evaluate[FilterOptions[Plot,  opts]]]}
]

(*
    This is needed so that the plot will have a nonzero
    width range even when one or the other coordinate
    has no change at all.  e.g. bug 51459 
*)
MakeRange[var_, vals_] := 
Module[{min, max},
    min = Min[vals];
    max = Max[vals];
    If[min == max,
        If[Developer`ZeroQ[min],
            min = -1; max = 1,
        (* else *)
            If[min > 0,
                min = 0; max = 2 max,
                max = 0; min = 2 min
            ]
        ]
    ];
    {var, min, max}
]

(*
 Messages
*)

FindMinimumProblem::unk = "The problem, `1`, is not known as a FindMinimum problem. The known problems for FindMinimum are listed by $FindMinimumProblems."

FindRootProblem::unk = "The problem, `1`, is not known as a FindRoot problem. The known problems for FindRoot are listed by $FindRootProblems."

FindMinimumProblem::size = "The size, `1`, is not legal for the problem, `2`. A legal size for this problem matches the pattern, `3`."

FindRootProblem::size = "The size, n = `1`, is not legal for the problem, `2`. With a legal size, {n, n} matches the pattern, `3`."

FindMinimumProblem::start = FindRootProblem::start = "The starting point, `1`, for the problem, `2`, with size {n, m} = `3` is not a list of length n."

FindRootProblem::start = "The starting point, `1`, for the problem, `2`, with size n = `3` is not a list of length n."

FindMinimumProblem::args = "The arguments, `1`, cannot be recognized as valid input."

ProblemTest::unkf = "An error in the specification of the problem, `1`, has been found."

UnconstrainedProblem::context = "The problem name `1` was used before loading this package and previous definitions may shadow its use here."

End[]

EndPackage[]






