(* :Copyright: Copyright 2003, Wolfram Research, Inc. *)

(* :Mathematica Version: 5.0 *)

(* :Title: Tools for working with URLs *)

(* :Author: Yifan Hu and Tom Wickham-Jones *)

(* :Keywords: FetchURL, UncompressGZIPFile *)


(* :Summary:
This package uses J/Link calls to copy files from a URL to a local file. It also 
provides a tool for unpacking GZIP files.
*)


BeginPackage["Utilities`URLTools`", "JLink`"]

FetchURL::usage = "FetchURL[ url, new] copies the file referenced by url to the file new on the local machine. FetchURL[ url] copies the file creating a new filename. FetchURL returns the name of the temporary file used on the local machine."

UncompressGZIPFile::usage = "UncompressGZIPFile[ old, new] unformats the gzip file old into new."

SetFTPProxy::usage = "SetFTPProxy[ host, port] sets the proxy server to use for FTP connections."

RemoveFTPProxy::usage = "RemoveFTPProxy[ ] removes any proxy servers for FTP connections."

FileFilters::usage = "FileFilters is an option of FetchURL that sets up pairs {patt, fun}. Each of these apply the function 'fun' to the file if the name matches patt. This can be used for decompressing files."

$UserAgent::usage = "$UserAgent is the default value for the user agent property of the connection."


Begin["`Private`"];


FetchURL::conopen = "The connection to `1` cannot be opened."


FetchURL::contime = 
    "The maximum connection time of `1` seconds has been exceeded.";

FetchURL::nofile = UncompressGZIPFile::nofile = "File `1` cannot be opened.";

UncompressGZIPFile::outfile = "File name `1` is not a string.";


$UserAgent = None;


$JavaUninstalled = True;

If[ 
	!ValueQ[Utilities`URLTools`Internal`$InstallJavaArgs], Utilities`URLTools`Internal`$InstallJavaArgs={}];

InstallJava @@ Utilities`URLTools`Internal`$InstallJavaArgs

DEBUG = False; MONITOR = False;


(* ftp proxy setup *)
ftpProxyHost = "proxy";
ftpProxyPort = "8080";

(* exceptions *)
{BADCONNECTION , CANTOPENOUTFILE, BADFILE, CONTIMEOUT, NOTSTRING}; 

(* IO/buffer size and max connection time for downloading a packet of this \
size *)
CONTIME = 5;
BUFSIZE = 5000;



(*
Create a temporary unique file name. If a suffix is given this attached to 
the file after three _ characters.
*)
      
createTempFile[ prefix_String] := createTempFile[ prefix, ""]

createTempFile[ prefix_String, suffix_String] :=
	JavaBlock[ 
		Module[ {f, ss},
			If[ $JavaUninstalled, InstallJava[];$JavaUninstalled = False];
			If[ StringLength[ suffix] === 0, ss = "", ss = "___" <> suffix];
			LoadClass[ "java.io.File"];
			f = File`createTempFile[ prefix, ss];
			f@getCanonicalPath[]]
	]


(*
 Returns the chars after the last '/' character.
*)
 
getFileName[ str_String] :=
	Module[ {pos},
      (* drop all except word characters (including _), and . *)
      pos = Complement[
          StringPosition[str, RegularExpression["\\W"]],
          StringPosition[str, "."]
      ];
		If[ pos === {}, 
			"unknown.dat",
			pos = pos[[-1,1]]; 
			StringDrop[ str, pos]]
	]

(*
 Process any exceptions that were thrown, issue messages as appropriate.
*)

processException[ _, URLToolsException[ except_, arg_]] :=
	(Switch[ except,
			BADCONNECTION, Message[FetchURL::conopen, arg],
			CONTIMEOUT, Message[FetchURL::contime, CONTIME],
			BADFILE, Message[UncompressGZIPFile::nofile, arg],
			CANTOPENOUTFILE, Message[FetchURL::nofile, arg],
			NOTSTRING, Message[UncompressGZIPFile::outfile, arg],
			_,1]; $Failed)



(*
 Implementation of FetchURL.
*)

Options[ FetchURL] = {FileFilters -> {{".gz", UncompressGZIPFile}}}


ValidFilter[ {{_String, _}..}] := True

ValidFilter[ ___] := False


(*
 If patt matches the end of name, then apply fun to the file. 
 Returning the new file name.  
*)

ApplyFilter[ {patt_String, fun_}, file_, nameIn_, autoName_] :=
	Module[ {fileOut = file, name = nameIn},
		If[ StringMatchQ[ file, "*" <> patt],
			name = StringDrop[ name, -StringLength[ patt]];
			fileOut = name;
			If[ autoName,
				fileOut = createTempFile[ "Temp", fileOut]];
			fileOut = fun[ file, fileOut]];	
			{fileOut, name}
	]


FetchURL[ url_String, outFileIn_String:"", opts___?OptionQ] :=
	Module[ {epil, file, outFile = outFileIn, name = outFileIn, autoName = False},
		If[ !StringQ[ outFile] || StringLength[outFile] < 1,
			autoName = True;
			name = getFileName[ url];
		 	outFile = createTempFile[ "Temp", name]];      	
		epil = FileFilters /. {opts} /. Options[ FetchURL] ;
		Catch[ 
			file = iFetchURL[ url, outFile];
			If[ ValidFilter[ epil],
				Map[ ({file, name} = ApplyFilter[ #, file, name, autoName])&, epil]];
			file
			,
			_URLToolsException,
			processException]
	]

 
(*Returns True if the pathname begins with a relative path metacharacter. 
  This is copied from Converteres.m*)
(*for MacOS*)

beginsRelativeMetaCharQ[
  str_String] := (StringMatchQ[str, ":"] || StringMatchQ[str, "::"] || 
    StringMatchQ[str, ToFileName[{":"}, "*"]] || 
    StringMatchQ[str, ToFileName[{"::"}, "*"]]) /; 
  StringMatchQ[$OperatingSystem, "MacOS"]
(*for non - MacOS*)

beginsRelativeMetaCharQ[str_String] := 
 StringMatchQ[str, "."] || StringMatchQ[str, ".."] || 
  StringMatchQ[str, ToFileName[{"."}, "*"]] || 
  StringMatchQ[str, 
   ToFileName[{".."}, 
    "*"]] ||(*hack to make sure full pathname on Japanese Windows*)($SystemID \
=== "Windows" && $Language === "Japanese" && StringTake[str, {2}] != ":")
beginsRelativeMetaCharQ[___] := False


(* set a file to working directory unless absolute path is specified *)

setToWorkingDirectory[file_] := Module[
  {ffile = file},
  If[beginsRelativeMetaCharQ[ffile] || DirectoryName[ffile] === "", 
   ffile = ToFileName[{Directory[]}, ffile]];
  ffile]

iFetchURL[url_String, outFile1_String] := JavaBlock[
	Module[
		{fos, t, u, is, buf, totalBytes = 0, flag, ff, outFile = outFile1, conn},
		If[ $JavaUninstalled, InstallJava[];$JavaUninstalled = False];

      (* make the file in the Mathemnatica working directory Directory[],
         unless the user specified another directory*)
      outFile = setToWorkingDirectory[outFile];
      
     	fos = JavaNew["java.io.FileOutputStream", outFile];
      If [fos === $Failed, 
            Throw[Null, URLToolsException[CANTOPENOUTFILE, outFile]]
      ];
		t = Timing[
			Internal`DeactivateMessages[
            	Check[
              		u = JavaNew["java.net.URL", url];
              		conn = u@openConnection[];
              		If[ StringQ[ $UserAgent],
	              		conn@setRequestProperty["User-Agent", $UserAgent]];
              		is = conn@getInputStream[],
              		is = $Failed
              		]
            ];
		If[is === $Failed,
            Throw[Null, URLToolsException[ BADCONNECTION, url]]
		];
          
		(* check connection first, 
			get 8 bytes. This does not work yet as once entering Java, 
			timeconstrainted worul not work. *)
          
		buf = JavaNew["[B", 8];
		flag = TimeConstrained[
				numRead = is@read[buf];
				totalBytes += numRead;
				fos@write[buf, 0, numRead], CONTIME, $Failed
				];
          
		If [flag === $Failed, 
            	Throw[Null, URLToolsException[ CONTIMEOUT, url]]
		];
          
		(* now read safely knowing that connection is OK *)
          
		buf = JavaNew["[B", BUFSIZE];
		While[(numRead = is@read[buf]) > 0,
			totalBytes += numRead;
            If[DEBUG, Print[totalBytes, " bytes downloaded"]];
            fos@write[buf, 0, numRead]
		];
		is@close[];
		fos@close[];
		];
      	If [MONITOR, 
        	Print["download time = ", t[[1]], " bytes = ", totalBytes]];
      	outFile
      	]];



(*
 Implementation of UncompressGZIPFile.
*)

UncompressGZIPFile[ file_, outFile_:Null] :=
		Catch[ If[ outFile === Null, iUncompressGZIPFile[ file], iUncompressGZIPFile[ file, outFile]],
			_URLToolsException,
			processException]
  
  
iUncompressGZIPFile[ file_] := iUncompressGZIPFile[ file, createTempFile[ "Temp"]]

  
iUncompressGZIPFile[file2_, outFile_] := JavaBlock[
Internal`DeactivateMessages[
	Module[
      	{t, fis, gzis, fos, totalBytes = 0, buf, outFile2 = outFile, file = file2},
		If[ $JavaUninstalled, InstallJava[];$JavaUninstalled = False];
		If[!StringQ[ outFile2],
            	Throw[Null, URLToolsException[ NOTSTRING, outFile2]]
			];
		If[!StringQ[file],
            	Throw[Null, URLToolsException[ NOTSTRING, file]]
			];

      (* make the file in the Mathemnatica working directory Directory[],
         unless the user specified another directory*)
      outFile2 = setToWorkingDirectory[outFile2];
      file = setToWorkingDirectory[file];

      	t = Timing[
			fis = JavaNew["java.io.FileInputStream", file];
			If[fis === $Failed,
            	Throw[Null, URLToolsException[ BADFILE, file]]
			];
          
			gzis = JavaNew["java.util.zip.GZIPInputStream", fis];
			fos = JavaNew["java.io.FileOutputStream", outFile2];
			buf = JavaNew["[B", BUFSIZE];
          
			While[(numRead = gzis@read[buf]) > 0,
				totalBytes += numRead;
            	If[DEBUG, Print[totalBytes, " bytes uncompressed and written"]];
            	fos@write[buf, 0, numRead]
			];
			fis@close[];
          	gzis@close[];
          	fos@close[];
		];
		If [MONITOR, 
			Print["uncompress time = ", t[[1]], " bytes = ", totalBytes];];
		outFile2
	]]];


(*
 FTP proxy utilities.
*)
      
SetFTPProxy[ftpProxyHost_, ftpProxyPortIn_] := Module[
      {sysProperties, ftpProxyPort = ftpProxyPortIn},
		If[ $JavaUninstalled, InstallJava[];$JavaUninstalled = False];
		If[ !StringQ[ ftpProxyPort], ftpProxyPort = ToString[ ftpProxyPort]];
      LoadClass["java.lang.System"];
      sysProperties = System`getProperties[];
      sysProperties@
        put[MakeJavaObject["ftpProxySet"], MakeJavaObject["true"]];
      sysProperties@
        put[MakeJavaObject["ftpProxyHost"], MakeJavaObject[ftpProxyHost]];
      sysProperties@
        put[MakeJavaObject["ftpProxyPort"], MakeJavaObject[ftpProxyPort]];
      ];
      
      
RemoveFTPProxy[] := Module[
      {sysProperties},
		If[ $JavaUninstalled, InstallJava[];$JavaUninstalled = False];
      LoadClass["java.lang.System"];
      sysProperties = System`getProperties[];
      sysProperties@remove[MakeJavaObject["ftpProxySet"]];
      sysProperties@remove[MakeJavaObject["ftpProxyHost"]];
      sysProperties@remove[MakeJavaObject["ftpProxyPort"]];
      ];
  	
	
	
 
End[]; (* end private *)
EndPackage[];
