(************** Content-type: application/mathematica **************
                     CreatedBy='Mathematica 5.2'

                    Mathematica-Compatible Notebook

This notebook can be used with any Mathematica-compatible
application, such as Mathematica, MathReader or Publicon. The data
for the notebook starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do
one of the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the
  application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing
the word CacheID, otherwise Mathematica-compatible applications may
try to use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
*******************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     38422,       1188]*)
(*NotebookOutlinePosition[     58734,       1811]*)
(*  CellTagsIndexPosition[     56391,       1713]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell[TextData[{
  StyleBox["1.12 ", "HeadingNumber"],
  "Special Topic: The Internals of ",
  StyleBox["Mathematica", "SBO"]
}], "Section",
  CellTags->{"S1.12.0", "0.1"}],

Cell[CellGroupData[{

Cell["1.12.1 Why You Do Not Usually Need to Know about Internals", \
"Subsection",
  CellTags->{"S1.12.1", "1.1"}],

Cell[TextData[{
  "Most of this book is concerned with explaining ",
  StyleBox["what", "TI"],
  " ",
  StyleBox["Mathematica", "TI"],
  " does, not ",
  StyleBox["how", "TI"],
  " it does it. But the purpose of this chapter is to say at least a little \
about how ",
  StyleBox["Mathematica", "TI"],
  " does what it does. Appendix ",
  ButtonBox["A.9",
    ButtonData:>{"A.9"},
    ButtonStyle->"MainBookLink"],
  " gives some more details. "
}], "Text",
  CellTags->{"S1.12.1", "1.2"}],

Cell[TextData[{
  "You should realize at the outset that while knowing about the internals of \
",
  StyleBox["Mathematica", "TI"],
  " may be of intellectual interest, it is usually much less important in \
practice than one might at first suppose. "
}], "Text",
  CellTags->{"S1.12.1", "1.3"}],

Cell[TextData[{
  "Indeed, one of the main points of ",
  StyleBox["Mathematica", "TI"],
  " is that it provides an environment where you can perform mathematical and \
other operations without having to think in detail about how these operations \
are actually carried out inside your computer. "
}], "Text",
  CellTags->{"S1.12.1", "1.4"}],

Cell[TextData[{
  "Thus, for example, if you want to factor the polynomial ",
  Cell[BoxData[
      FormBox[
        RowBox[{
          SuperscriptBox[
            StyleBox["x",
              "TI"], "15"], "-", "1"}], TraditionalForm]], "InlineFormula"],
  ", you can do this just by giving ",
  StyleBox["Mathematica", "TI"],
  " the command ",
  StyleBox["Factor[x^15", "MR"],
  " ",
  StyleBox["-", "MR"],
  " ",
  StyleBox["1]", "MR"],
  "; you do not have to know the fairly complicated details of how such a \
factorization is actually carried out by the internal code of ",
  StyleBox["Mathematica", "TI"],
  ". "
}], "Text",
  CellTags->{"S1.12.1", "1.5"}],

Cell[TextData[{
  "Indeed, in almost all practical uses of ",
  StyleBox["Mathematica", "TI"],
  ", issues about how ",
  StyleBox["Mathematica", "TI"],
  " works inside turn out to be largely irrelevant. For most purposes it \
suffices to view ",
  StyleBox["Mathematica", "TI"],
  " simply as an abstract system which performs certain specified \
mathematical and other operations. "
}], "Text",
  CellTags->{"S1.12.1", "1.6"}],

Cell[TextData[{
  "You might think that knowing how ",
  StyleBox["Mathematica", "TI"],
  " works inside would be necessary in determining what answers it will give. \
But this is only very rarely the case. For the vast majority of the \
computations that ",
  StyleBox["Mathematica", "TI"],
  " does are completely specified by the definitions of mathematical or other \
operations. "
}], "Text",
  CellTags->{"S1.12.1", "1.7"}],

Cell[TextData[{
  "Thus, for example, ",
  StyleBox["3^40", "MR"],
  " will always be ",
  StyleBox["12157665459056928801", "MR"],
  ", regardless of how ",
  StyleBox["Mathematica", "TI"],
  " internally computes this result. "
}], "Text",
  CellTags->{"S1.12.1", "1.8"}],

Cell[TextData[{
  "There are some situations, however, where several different answers are \
all equally consistent with the formal mathematical definitions. Thus, for \
example, in computing symbolic integrals, there are often several different \
expressions which all yield the same derivative. Which of these expressions \
is actually generated by ",
  StyleBox["Integrate", "MR"],
  " can then depend on how ",
  StyleBox["Integrate", "MR"],
  " works inside. "
}], "Text",
  CellTags->{"S1.12.1", "1.9"}],

Cell[TextData[{
  "Here is the answer generated by ",
  StyleBox["Integrate", "MR"],
  ". "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.10"}],

Cell[CellGroupData[{

Cell["Integrate[1/x + 1/x^2, x]", "Input",
  CellLabel->"In[1]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(\(-\(1\/x\)\) + Log[x]\)], "Output",
  CellLabel->"Out[1]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell[TextData[{
  "This is an equivalent expression that might have been generated if ",
  StyleBox["Integrate", "MR"],
  " worked differently inside. "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.11"}],

Cell[CellGroupData[{

Cell["Together[%]", "Input",
  CellLabel->"In[2]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(\(\(-1\) + x\ Log[x]\)\/x\)], "Output",
  CellLabel->"Out[2]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell[TextData[{
  "In numerical computations, a similar phenomenon occurs. Thus, for example, \
",
  StyleBox["FindRoot", "MR"],
  " gives you a root of a function. But if there are several roots, which \
root is actually returned depends on the details of how ",
  StyleBox["FindRoot", "MR"],
  " works inside. "
}], "Text",
  CellTags->{"S1.12.1", "1.12"}],

Cell[TextData[{
  "This finds a particular root of ",
  Cell[BoxData[
      FormBox[
        RowBox[{"cos", 
          RowBox[{"(", 
            StyleBox["x",
              "TI"], ")"}], "+", "sin", 
          RowBox[{"(", 
            StyleBox["x",
              "TI"], ")"}]}], TraditionalForm]], "InlineFormula"],
  ". "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.13"}],

Cell[CellGroupData[{

Cell["FindRoot[Cos[x] + Sin[x], {x, 10.5}]", "Input",
  CellLabel->"In[3]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \({x \[Rule] 14.922565104551518`}\)], "Output",
  CellLabel->"Out[3]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell["\<\
With a different starting point, a different root is found. Which \
root is found with each starting point depends in detail on the internal \
algorithm used. \
\>", "MathCaption",
  CellTags->{"S1.12.1", "1.14"}],

Cell[CellGroupData[{

Cell["FindRoot[Cos[x] + Sin[x], {x, 10.8}]", "Input",
  CellLabel->"In[4]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \({x \[Rule] 11.780972450961725`}\)], "Output",
  CellLabel->"Out[4]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell["\<\
The dependence on the details of internal algorithms can be more \
significant if you push approximate numerical computations to the limits of \
their validity. \
\>", "Text",
  CellTags->{"S1.12.1", "1.15"}],

Cell[TextData[{
  "Thus, for example, if you give ",
  StyleBox["NIntegrate", "MR"],
  " a pathological integrand, whether it yields a meaningful answer or not \
can depend on the details of the internal algorithm that it uses. "
}], "Text",
  CellTags->{"S1.12.1", "1.16"}],

Cell[TextData[{
  StyleBox["NIntegrate", "MR"],
  " knows that this result is unreliable, and can depend on the details of \
the internal algorithm, so it prints warning messages. "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.17"}],

Cell[CellGroupData[{

Cell["NIntegrate[Sin[1/x], {x, 0, 1}]", "Input",
  CellLabel->"In[5]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(NIntegrate::"slwcon" \(\(:\)\(\ \)\) 
      "Numerical integration converging too slowly; suspect one of the \
following: singularity, value of the integration being 0, oscillatory \
integrand, or insufficient WorkingPrecision. If your integrand is oscillatory \
try using the option Method->Oscillatory in NIntegrate."\)], "Message",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(NIntegrate::"ncvb" \(\(:\)\(\ \)\) 
      "NIntegrate failed to converge to prescribed accuracy after \!\(7\) \
recursive bisections in \!\(x\) near \!\(x\) = \!\(0.0035126778890767337`\)."\
\)], "Message",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(0.5048942747035711`\)], "Output",
  CellLabel->"Out[5]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell["\<\
Traditional numerical computation systems have tended to follow the \
idea that all computations should yield results that at least nominally have \
the same precision. A consequence of this idea is that it is not sufficient \
just to look at a result to know whether it is accurate; you typically also \
have to analyze the internal algorithm by which the result was found. This \
fact has tended to make people believe that it is always important to know \
internal algorithms for numerical computations. \
\>", "Text",
  CellTags->{"S1.12.1", "1.18"}],

Cell[TextData[{
  "But with the approach that ",
  StyleBox["Mathematica", "TI"],
  " takes, this is rarely the case. For ",
  StyleBox["Mathematica", "TI"],
  " can usually use its arbitrary\[Hyphen]precision numerical computation \
capabilities to give results where every digit that is generated follows the \
exact mathematical specification of the operation being performed. "
}], "Text",
  CellTags->{"S1.12.1", "1.19"}],

Cell[TextData[{
  "Even though this is an approximate numerical computation, every digit is \
determined by the mathematical definition for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Pi]\)], "InlineFormula"],
  ". "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.20"}],

Cell[CellGroupData[{

Cell["N[Pi, 30]", "Input",
  CellLabel->"In[6]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(3.141592653589793238462643383279502884197167382228`30. \)], "Output",
  CellLabel->"Out[6]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell[TextData[{
  "Once again, every digit here is determined by the mathematical definition \
for ",
  Cell[BoxData[
      FormBox[
        RowBox[{"sin", 
          RowBox[{"(", 
            StyleBox["x",
              "TI"], ")"}]}], TraditionalForm]], "InlineFormula"],
  ". "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.21"}],

Cell[CellGroupData[{

Cell["N[Sin[10^50], 20]", "Input",
  CellLabel->"In[7]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(\(-0.78967249342931008271028953991740775626`20. \)\)], "Output",
  CellLabel->"Out[7]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell[TextData[{
  "If you use machine\[Hyphen]precision numbers, ",
  StyleBox["Mathematica", "TI"],
  " cannot give a reliable result, and the answer depends on the details of \
the internal algorithm used. "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.22"}],

Cell[CellGroupData[{

Cell["Sin[10.^50]", "Input",
  CellLabel->"In[8]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(0.7052215258133128`\)], "Output",
  CellLabel->"Out[8]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell[TextData[{
  "It is a general characteristic that whenever the results you get can be \
affected by the details of internal algorithms, you should not depend on \
these results. For if nothing else, different versions of ",
  StyleBox["Mathematica", "TI"],
  " may exhibit differences in these results, either because the algorithms \
operate slightly differently on different computer systems, or because \
fundamentally different algorithms are used in versions released at different \
times. "
}], "Text",
  CellTags->{"S1.12.1", "1.23"}],

Cell[TextData[{
  "This is the result for ",
  Cell[BoxData[
      \(TraditionalForm\`sin(10\^50)\)], "InlineFormula"],
  " on one type of computer. "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.24"}],

Cell[CellGroupData[{

Cell["Sin[10.^50]", "Input",
  CellLabel->"In[1]:= ",
  CellTags->"S1.12.1"],

Cell[BoxData[
    \(0.7052215258133128`\)], "Output",
  CellLabel->"Out[1]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell["Here is the same calculation on another type of computer. ", \
"MathCaption",
  CellTags->{"S1.12.1", "1.25"}],

Cell[CellGroupData[{

Cell["Sin[10.^50]", "Input",
  CellLabel->"In[1]:= ",
  CellTags->"S1.12.1"],

Cell["-0.0528229", "Output",
  CellLabel->"Out[1]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell[TextData[{
  "And here is the result obtained in ",
  StyleBox["Mathematica", "TI"],
  " Version 1. "
}], "MathCaption",
  CellTags->{"S1.12.1", "1.26"}],

Cell[CellGroupData[{

Cell["Sin[10.^50]", "Input",
  CellLabel->"In[2]:= ",
  CellTags->"S1.12.1"],

Cell["0.0937538", "Output",
  CellLabel->"Out[2]= ",
  CellTags->"S1.12.1"]
}, Open  ]],

Cell[TextData[{
  "Particularly in more advanced applications of ",
  StyleBox["Mathematica", "TI"],
  ", it may sometimes seem worthwhile to try to analyze internal algorithms \
in order to predict which way of doing a given computation will be the most \
efficient. And there are indeed occasionally major improvements that you will \
be able to make in specific computations as a result of such analyses.   "
}], "Text",
  CellTags->{"S1.12.1", "1.27"}],

Cell[TextData[{
  "But most often the analyses will not be worthwhile. For the internals of \
",
  StyleBox["Mathematica", "TI"],
  " are quite complicated, and even given a basic description of the \
algorithm used for a particular purpose, it is usually extremely difficult to \
reach a reliable conclusion about how the detailed implementation of this \
algorithm will actually behave in particular circumstances. "
}], "Text",
  CellTags->{"S1.12.1", "1.28"}],

Cell[TextData[{
  "A typical problem is that ",
  StyleBox["Mathematica", "TI"],
  " has many internal optimizations, and the efficiency of a computation can \
be greatly affected by whether the details of the computation do or do not \
allow a given internal optimization to be used. "
}], "Text",
  CellTags->{"S1.12.1", "1.29"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["1.12.2 Basic Internal Architecture", "Subsection",
  CellTags->{"S1.12.2", "2.1"}],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell["numbers "], Cell["sequences of binary digits "]},
          {Cell["strings "], Cell[
            "sequences of character code bytes or byte pairs "]},
          {Cell["symbols "], Cell[
            "pointers to the central table of symbols "]},
          {Cell["general expressions "], Cell[
            "sequences of pointers to the head and elements "]}
          },
        ColumnAlignments->{Right, Left}], TraditionalForm]], "DefinitionBox",
  GridBoxOptions->{RowSpacings->0.4,
  RowLines->False},
  CellTags->"S1.12.2"],

Cell[TextData[{
  "Internal representations used by ",
  StyleBox["Mathematica", "SO"],
  ". "
}], "Caption",
  CellTags->"S1.12.2"],

Cell[TextData[{
  "When you type input into ",
  StyleBox["Mathematica", "TI"],
  ", a data structure is created in the memory of your computer to represent \
the expression you have entered. "
}], "Text",
  CellTags->{"S1.12.2", "2.2"}],

Cell[TextData[{
  "In general, different pieces of your expression will be stored at \
different places in memory. Thus, for example, for a list such as ",
  StyleBox["{2,", "MR"],
  " ",
  StyleBox["x,", "MR"],
  " ",
  StyleBox["y", "MR"],
  " ",
  StyleBox["+", "MR"],
  " ",
  StyleBox["z}", "MR"],
  " the \[OpenCurlyDoubleQuote]backbone\[CloseCurlyDoubleQuote] of the list \
will be stored at one place, while each of the actual elements will be stored \
at a different place. "
}], "Text",
  CellTags->{"S1.12.2", "2.3"}],

Cell["\<\
The backbone of the list then consists just of three \
\[OpenCurlyDoubleQuote]pointers\[CloseCurlyDoubleQuote] that specify the \
addresses in computer memory at which the actual expressions that form the \
elements of the list are to be found. These expressions then in turn contain \
pointers to their subexpressions. The chain of pointers ends when one reaches \
an object such as a number or a string, which is stored directly as a pattern \
of bits in computer memory. \
\>", "Text",
  CellTags->{"S1.12.2", "2.4"}],

Cell[TextData[{
  "Crucial to the operation of ",
  StyleBox["Mathematica", "TI"],
  " is the notion of symbols such as ",
  StyleBox["x", "MR"],
  ". Whenever ",
  StyleBox["x", "MR"],
  " appears in an expression, ",
  StyleBox["Mathematica", "TI"],
  " represents it by a pointer. But the pointer is always to the same place \
in computer memory\[LongDash]an entry in a central table of all symbols \
defined in your ",
  StyleBox["Mathematica", "TI"],
  " session.  "
}], "Text",
  CellTags->{"S1.12.2", "2.5"}],

Cell["\<\
This table is a repository of all information about each symbol. It \
contains a pointer to a string giving the symbol's name, as well as pointers \
to expressions which give rules for evaluating the symbol. \
\>", "Text",
  CellTags->{"S1.12.2", "2.6"}],

Cell[BoxData[
    FormBox[GridBox[{
          {
            Cell["\<\
\[FilledSmallSquare] Recycle memory as soon as the data \
in it is no longer referenced. \
\>"]}
          },
        ColumnAlignments->{Left}], TraditionalForm]], "FramedBox",
  GridBoxOptions->{RowSpacings->0.4,
  RowLines->False},
  CellTags->"S1.12.2"],

Cell[TextData[{
  "The basic principle of ",
  StyleBox["Mathematica", "SO"],
  " memory management. "
}], "Caption",
  CellTags->"S1.12.2"],

Cell[TextData[{
  "Every piece of memory used by ",
  StyleBox["Mathematica", "TI"],
  " maintains a count of how many pointers currently point to it. When this \
count drops to zero, ",
  StyleBox["Mathematica", "TI"],
  " knows that the piece of memory is no longer being referenced, and \
immediately makes the piece of memory available for something new. "
}], "Text",
  CellTags->{"S1.12.2", "2.7"}],

Cell[TextData[{
  "This strategy essentially ensures that no memory is ever wasted, and that \
any piece of memory that ",
  StyleBox["Mathematica", "TI"],
  " uses is actually storing data that you need to access in your ",
  StyleBox["Mathematica", "TI"],
  " session. "
}], "Text",
  CellTags->{"S1.12.2", "2.8"}],

Cell[BoxData[
    FormBox[GridBox[{
          {
            Cell["\<\
\[FilledSmallSquare] Create an expression corresponding \
to the input you have given. \
\>"]},
          {
            Cell["\<\
\[FilledSmallSquare] Process the expression using all \
rules known for the objects in it. \
\>"]},
          {
            Cell["\<\
\[FilledSmallSquare] Generate output corresponding to \
the resulting expression. \
\>"]}
          },
        ColumnAlignments->{Left}], TraditionalForm]], "FramedBox",
  GridBoxOptions->{RowSpacings->0.4,
  RowLines->False},
  CellTags->"S1.12.2"],

Cell[TextData[{
  "The basic actions of ",
  StyleBox["Mathematica", "SO"],
  ". "
}], "Caption",
  CellTags->"S1.12.2"],

Cell[TextData[{
  "At the heart of ",
  StyleBox["Mathematica", "TI"],
  " is a conceptually simple procedure known as the ",
  StyleBox["evaluator", "TI"],
  " which takes every function that appears in an expression and evaluates \
that function. "
}], "Text",
  CellTags->{"S1.12.2", "2.9"}],

Cell[TextData[{
  "When the function is one of the thousand or so that are built into ",
  StyleBox["Mathematica", "TI"],
  ", what the evaluator does is to execute directly internal code in the ",
  StyleBox["Mathematica", "TI"],
  " system. This code is set up to perform the operations corresponding to \
the function, and then to build a new expression representing the result. "
}], "Text",
  CellTags->{"S1.12.2", "2.10"}],

Cell[BoxData[
    FormBox[GridBox[{
          {Cell[TextData[{
              "\[FilledSmallSquare] The built\[Hyphen]in functions of ",
              StyleBox["Mathematica", "TI"],
              " support universal computation. "
            }]]}
          },
        ColumnAlignments->{Left}], TraditionalForm]], "FramedBox",
  GridBoxOptions->{RowSpacings->0.4,
  RowLines->False},
  CellTags->"S1.12.2"],

Cell[TextData[{
  "The basic feature that makes ",
  StyleBox["Mathematica", "SO"],
  " a self\[Hyphen]contained system. "
}], "Caption",
  CellTags->"S1.12.2"],

Cell[TextData[{
  "A crucial feature of the built\[Hyphen]in functions in ",
  StyleBox["Mathematica", "TI"],
  " is that they support ",
  StyleBox["universal computation", "TI"],
  ". What this means is that out of these functions you can construct \
programs that perform absolutely any kinds of operation that are possible for \
a computer. "
}], "Text",
  CellTags->{"S1.12.2", "2.11"}],

Cell[TextData[{
  "As it turns out, small subsets of ",
  StyleBox["Mathematica", "TI"],
  "'s built\[Hyphen]in functions would be quite sufficient to support \
universal computation. But having the whole collection of functions makes it \
in practice easier to construct the programs one needs. "
}], "Text",
  CellTags->{"S1.12.2", "2.12"}],

Cell[TextData[{
  "The underlying point, however, is that because ",
  StyleBox["Mathematica", "TI"],
  " supports universal computation you never have to modify its \
built\[Hyphen]in functions: all you have to do to perform a particular task \
is to combine these functions in an appropriate way. "
}], "Text",
  CellTags->{"S1.12.2", "2.13"}],

Cell[TextData[{
  "Universal computation is the basis for all standard computer languages. \
But many of these languages rely on the idea of ",
  StyleBox["compilation", "TI"],
  ". If you use C or Fortran, for example, you first write your program, then \
you compile it to generate machine code that can actually be executed on your \
computer.  "
}], "Text",
  CellTags->{"S1.12.2", "2.14"}],

Cell[TextData[{
  StyleBox["Mathematica", "TI"],
  " does not require you to go through the compilation step: once you have \
input an expression, the functions in the expression can immediately be \
executed. "
}], "Text",
  CellTags->{"S1.12.2", "2.15"}],

Cell[TextData[{
  "Often ",
  StyleBox["Mathematica", "TI"],
  " will preprocess expressions that you enter, arranging things so that \
subsequent execution will be as efficient as possible. But such preprocessing \
never affects the results that are generated, and can rarely be seen \
explicitly. "
}], "Text",
  CellTags->{"S1.12.2", "2.16"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "1.12.3 The Algorithms of ",
  StyleBox["Mathematica", "SBO"]
}], "Subsection",
  CellTags->{"S1.12.3", "3.1"}],

Cell[TextData[{
  "The built\[Hyphen]in functions of ",
  StyleBox["Mathematica", "TI"],
  " implement a very large number of algorithms from computer science and \
mathematics. Some of these algorithms are fairly old, but the vast majority \
had to be created or at least modified specifically for ",
  StyleBox["Mathematica", "TI"],
  ". Most of the more mathematical algorithms in ",
  StyleBox["Mathematica", "TI"],
  " ultimately carry out operations which at least at some time in the past \
were performed by hand. In almost all cases, however, the algorithms use \
methods very different from those common in hand calculation. "
}], "Text",
  CellTags->{"S1.12.3", "3.2"}],

Cell["\<\
Symbolic integration provides an example. In hand calculation, \
symbolic integration is typically done by a large number of tricks involving \
changes of variables and the like. \
\>", "Text",
  CellTags->{"S1.12.3", "3.3"}],

Cell[TextData[{
  "But in ",
  StyleBox["Mathematica", "TI"],
  " symbolic integration is performed by a fairly small number of very \
systematic procedures. For indefinite integration, the idea of these \
procedures is to find the most general form of the integral, then to \
differentiate this and try to match up undetermined coefficients. "
}], "Text",
  CellTags->{"S1.12.3", "3.4"}],

Cell["\<\
Often this procedure produces at an intermediate stage immensely \
complicated algebraic expressions, and sometimes very sophisticated kinds of \
mathematical functions. But the great advantage of the procedure is that it \
is completely systematic, and its operation requires no special cleverness of \
the kind that only a human could be expected to provide. \
\>", "Text",
  CellTags->{"S1.12.3", "3.5"}],

Cell[TextData[{
  "In having ",
  StyleBox["Mathematica", "TI"],
  " do integrals, therefore, one can be confident that it will systematically \
get results, but one cannot expect that the way these results are derived \
will have much at all to do with the way they would be derived by hand. "
}], "Text",
  CellTags->{"S1.12.3", "3.6"}],

Cell[TextData[{
  "The same is true with most of the mathematical algorithms in ",
  StyleBox["Mathematica", "TI"],
  ". One striking feature is that even for operations that are simple to \
describe, the systematic algorithms to perform these operations in ",
  StyleBox["Mathematica", "TI"],
  " involve fairly advanced mathematical or computational ideas. "
}], "Text",
  CellTags->{"S1.12.3", "3.7"}],

Cell[TextData[{
  "Thus, for example, factoring a polynomial in ",
  Cell[BoxData[
      FormBox[
        StyleBox["x",
          "TI"], TraditionalForm]], "InlineFormula"],
  " is first done modulo a prime such as 17 by finding the null space of a \
matrix obtained by reducing high powers of ",
  Cell[BoxData[
      FormBox[
        StyleBox["x",
          "TI"], TraditionalForm]], "InlineFormula"],
  " modulo the prime and the original polynomial. Then factorization over the \
integers is achieved by \
\[OpenCurlyDoubleQuote]lifting\[CloseCurlyDoubleQuote] modulo successive \
powers of the prime using a collection of intricate theorems in algebra and \
analysis.  "
}], "Text",
  CellTags->{"S1.12.3", "3.8"}],

Cell[TextData[{
  "The use of powerful systematic algorithms is important in making the built\
\[Hyphen]in functions in ",
  StyleBox["Mathematica", "TI"],
  " able to handle difficult and general cases. But for easy cases that may \
be fairly common in practice it is often possible to use simpler and more \
efficient algorithms. "
}], "Text",
  CellTags->{"S1.12.3", "3.9"}],

Cell[TextData[{
  "As a result, built\[Hyphen]in functions in ",
  StyleBox["Mathematica", "TI"],
  " often have large numbers of extra pieces that handle various kinds of \
special cases. These extra pieces can contribute greatly to the complexity of \
the internal code, often taking what would otherwise be a five\[Hyphen]page \
algorithm and making it hundreds of pages long. "
}], "Text",
  CellTags->{"S1.12.3", "3.10"}],

Cell[TextData[{
  "Most of the algorithms in ",
  StyleBox["Mathematica", "TI"],
  ", including all their special cases, were explicitly constructed by hand. \
But some algorithms were instead effectively created automatically by \
computer. "
}], "Text",
  CellTags->{"S1.12.3", "3.11"}],

Cell["\<\
Many of the algorithms used for machine\[Hyphen]precision numerical \
evaluation of mathematical functions are examples. The main parts of such \
algorithms are formulas which are as short as possible but which yield the \
best numerical approximations. \
\>", "Text",
  CellTags->{"S1.12.3", "3.12"}],

Cell[TextData[{
  "Most such formulas used in ",
  StyleBox["Mathematica", "TI"],
  " were actually derived by ",
  StyleBox["Mathematica", "TI"],
  " itself. Often many months of computation were required, but the result \
was a short formula that can be used to evaluate functions in an optimal way. \
"
}], "Text",
  CellTags->{"S1.12.3", "3.13"}]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "1.12.4 The Software Engineering of ",
  StyleBox["Mathematica", "SBO"]
}], "Subsection",
  CellTags->{"S1.12.4", "4.1"}],

Cell[TextData[{
  StyleBox["Mathematica", "TI"],
  " is one of the more complex software systems ever constructed. Its source \
code is written in a combination of C and ",
  StyleBox["Mathematica", "TI"],
  ", and for Version 5, the code for the kernel consists of about 1.5 million \
lines of C and 150,000 lines of ",
  StyleBox["Mathematica", "TI"],
  ". This corresponds to roughly 50 megabytes of data, or some 50,000 printed \
pages.  "
}], "Text",
  CellTags->{"S1.12.4", "4.2"}],

Cell[TextData[{
  "The C code in ",
  StyleBox["Mathematica", "TI"],
  " is actually written in a custom extension of C which supports certain \
memory management and object\[Hyphen]oriented features. The ",
  StyleBox["Mathematica", "TI"],
  " code is optimized using ",
  StyleBox["Share", "MR"],
  " and ",
  StyleBox["DumpSave", "MR"],
  ". "
}], "Text",
  CellTags->{"S1.12.4", "4.3"}],

Cell[TextData[{
  "In the ",
  StyleBox["Mathematica", "TI"],
  " kernel the breakdown of different parts of the code is roughly as \
follows: language and system: 30%; numerical computation: 25%; algebraic \
computation: 25%; graphics and kernel output: 20%. "
}], "Text",
  CellTags->{"S1.12.4", "4.4"}],

Cell[TextData[{
  "Most of this code is fairly dense and algorithmic: those parts that are in \
effect simple procedures or tables use minimal code since they tend to be \
written at a higher level\[LongDash]often directly in ",
  StyleBox["Mathematica", "TI"],
  ". "
}], "Text",
  CellTags->{"S1.12.4", "4.5"}],

Cell[TextData[{
  "The source code for the kernel, save a fraction of a percent, is identical \
for all computer systems on which ",
  StyleBox["Mathematica", "TI"],
  " runs. "
}], "Text",
  CellTags->{"S1.12.4", "4.6"}],

Cell["\<\
For the front end, however, a significant amount of specialized \
code is needed to support each different type of user interface environment. \
The front end contains about 650,000 lines of system\[Hyphen]independent C \
source code, of which roughly 150,000 lines are concerned with expression \
formatting. Then there are between 50,000 and 100,000 lines of specific code \
customized for each user interface environment. \
\>", "Text",
  CellTags->{"S1.12.4", "4.7"}],

Cell[TextData[{
  StyleBox["Mathematica", "TI"],
  " uses a client\[Hyphen]server model of computing. The front end and kernel \
are connected via ",
  StyleBox["MathLink", "TI"],
  "\[LongDash]the same system as is used to communicate with other programs. \
"
}], "Text",
  CellTags->{"S1.12.4", "4.8"}],

Cell[TextData[{
  "Within the C code portion of the ",
  StyleBox["Mathematica", "TI"],
  " kernel, modularity and consistency are achieved by having different parts \
communicate primarily by exchanging complete ",
  StyleBox["Mathematica", "TI"],
  " expressions. "
}], "Text",
  CellTags->{"S1.12.4", "4.9"}],

Cell[TextData[{
  "But it should be noted that even though different parts of the system are \
quite independent at the level of source code, they have many algorithmic \
interdependencies. Thus, for example, it is common for numerical functions to \
make extensive use of algebraic algorithms, or for graphics code to use \
fairly advanced mathematical algorithms embodied in quite different ",
  StyleBox["Mathematica", "TI"],
  " functions. "
}], "Text",
  CellTags->{"S1.12.4", "4.10"}],

Cell[TextData[{
  "Since the beginning of its development in 1986, the effort spent directly \
on creating the source code for ",
  StyleBox["Mathematica", "TI"],
  " is a substantial fraction of a thousand man\[Hyphen]years. In addition, a \
comparable or somewhat larger effort has been spent on testing and \
verification. "
}], "Text",
  CellTags->{"S1.12.4", "4.11"}],

Cell[TextData[{
  "The source code of ",
  StyleBox["Mathematica", "TI"],
  " has changed greatly since Version 1 was released. The total number of \
lines of code in the kernel grew from 150,000 in Version 1 to 350,000 in \
Version 2, 600,000 in Version 3, 800,000 in Version 4 and about 1.5 million \
in Version 5. In addition, at every stage existing code has been revised\
\[LongDash]so that Version 5 has only a few percent of its code in common \
with Version 1. "
}], "Text",
  CellTags->{"S1.12.4", "4.12"}],

Cell[TextData[{
  "Despite these changes in internal code, however, the user\[Hyphen]level \
design of ",
  StyleBox["Mathematica", "TI"],
  " has remained compatible from Version 1 on. Much functionality has been \
added, but programs created for ",
  StyleBox["Mathematica", "TI"],
  " Version 1 will almost always run absolutely unchanged under Version 5. "
}], "Text",
  CellTags->{"S1.12.4", "4.13"}]
}, Open  ]],

Cell[CellGroupData[{

Cell["1.12.5 Testing and Verification", "Subsection",
  CellTags->{"S1.12.5", "5.1"}],

Cell[TextData[{
  "Every version of ",
  StyleBox["Mathematica", "TI"],
  " is subjected to a large amount of testing before it is released. The vast \
majority of this testing is done by an automated system that is written in ",
  StyleBox["Mathematica", "TI"],
  ". "
}], "Text",
  CellTags->{"S1.12.5", "5.2"}],

Cell[TextData[{
  "The automated system feeds millions of pieces of input to ",
  StyleBox["Mathematica", "TI"],
  ", and checks that the output obtained from them is correct. Often there is \
some subtlety in doing such checking: one must account for different behavior \
of randomized algorithms and for such issues as differences in machine\
\[Hyphen]precision arithmetic on different computers. "
}], "Text",
  CellTags->{"S1.12.5", "5.3"}],

Cell["\<\
The test inputs used by the automated system are obtained in \
several ways: \
\>", "Text",
  CellTags->{"S1.12.5", "5.4"}],

Cell[TextData[{
  "\[FilledSmallSquare] For every ",
  StyleBox["Mathematica", "TI"],
  " function, inputs are devised that exercise both common and extreme cases. \
 \n\[FilledSmallSquare] Inputs are devised to exercise each feature of the \
internal code.  \n\[FilledSmallSquare] All the examples in this book and in \
other books about ",
  StyleBox["Mathematica", "TI"],
  " are used.  \n\[FilledSmallSquare] Standard numerical tables are optically \
scanned for test inputs.  \n\[FilledSmallSquare] Formulas from all standard \
mathematical tables are entered.  \n\[FilledSmallSquare] Exercises from \
textbooks are entered.  \n\[FilledSmallSquare] For pairs of functions such as \
",
  StyleBox["Integrate", "MR"],
  " and ",
  StyleBox["D", "MR"],
  " or ",
  StyleBox["Factor", "MR"],
  " and ",
  StyleBox["Expand", "MR"],
  ", random expressions are generated and tested.  "
}], "Notes",
  CellTags->{"S1.12.5", "5.5"}],

Cell["\<\
When tests are run, the automated testing system checks not only \
the results, but also side effects such as messages, as well as memory usage \
and speed. \
\>", "Text",
  CellTags->{"S1.12.5", "5.6"}],

Cell[TextData[{
  "There is also a special instrumented version of ",
  StyleBox["Mathematica", "TI"],
  " which is set up to perform internal consistency tests. This version of ",
  StyleBox["Mathematica", "TI"],
  " runs at a small fraction of the speed of the real ",
  StyleBox["Mathematica", "TI"],
  ", but at every step it checks internal memory consistency, \
interruptibility, and so on. "
}], "Text",
  CellTags->{"S1.12.5", "5.7"}],

Cell[TextData[{
  "The instrumented version of ",
  StyleBox["Mathematica", "TI"],
  " also records which pieces of ",
  StyleBox["Mathematica", "TI"],
  " source code have been accessed, allowing one to confirm that all of the \
various internal functions in ",
  StyleBox["Mathematica", "TI"],
  " have been exercised by the tests given. "
}], "Text",
  CellTags->{"S1.12.5", "5.8"}],

Cell[TextData[{
  "All standard ",
  StyleBox["Mathematica", "TI"],
  " tests are routinely run on each version of ",
  StyleBox["Mathematica", "TI"],
  ", on each different computer system. Depending on the speed of the \
computer system, these tests take a few days to a few weeks of computer time. \
"
}], "Text",
  CellTags->{"S1.12.5", "5.9"}],

Cell["\<\
In addition, huge numbers of tests based on random inputs are run \
for the equivalent of many years of computer time on a sampling of different \
computer systems. \
\>", "Text",
  CellTags->{"S1.12.5", "5.10"}],

Cell[TextData[{
  "Even with all this testing, however, it is inevitable in a system as \
complex as ",
  StyleBox["Mathematica", "TI"],
  " that errors will remain. "
}], "Text",
  CellTags->{"S1.12.5", "5.11"}],

Cell[TextData[{
  "The standards of correctness for ",
  StyleBox["Mathematica", "TI"],
  " are certainly much higher than for typical mathematical proofs. But just \
as long proofs will inevitably contain errors that go undetected for many \
years, so also a complex software system such as ",
  StyleBox["Mathematica", "TI"],
  " will contain errors that go undetected even after millions of people have \
used it. "
}], "Text",
  CellTags->{"S1.12.5", "5.12"}],

Cell[TextData[{
  "Nevertheless, particularly after all the testing that has been done on it, \
the probability that you will actually discover an error in ",
  StyleBox["Mathematica", "TI"],
  " in the course of your work is extremely low. "
}], "Text",
  CellTags->{"S1.12.5", "5.13"}],

Cell[TextData[{
  "Doubtless there will be times when ",
  StyleBox["Mathematica", "TI"],
  " does things you do not expect. But you should realize that the \
probabilities are such that it is vastly more likely that there is something \
wrong with your input to ",
  StyleBox["Mathematica", "TI"],
  " or your understanding of what is happening than with the internal code of \
the ",
  StyleBox["Mathematica", "TI"],
  " system itself. "
}], "Text",
  CellTags->{"S1.12.5", "5.14"}],

Cell[TextData[{
  "If you do believe that you have found a genuine error in ",
  StyleBox["Mathematica", "TI"],
  ", then you should contact Wolfram Research Technical Support, so that the \
error can be corrected in future versions. "
}], "Text",
  CellTags->{"S1.12.5", "5.15"}]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"5.2 for X",
ScreenRectangle->{{0, 1024}, {0, 768}},
WindowSize->{520, 600},
WindowMargins->{{252, Automatic}, {Automatic, 84}},
StyleDefinitions -> "HelpBrowser.nb"
]

(*******************************************************************
Cached data follows.  If you edit this Notebook file directly, not
using Mathematica, you must remove the line containing CacheID at
the top of  the file.  The cache data will then be recreated when
you save this file from within Mathematica.
*******************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "S1.12.0"->{
    Cell[1776, 53, 171, 5, 91, "Section",
      CellTags->{"S1.12.0", "0.1"}]},
  "0.1"->{
    Cell[1776, 53, 171, 5, 91, "Section",
      CellTags->{"S1.12.0", "0.1"}]},
  "S1.12.1"->{
    Cell[1972, 62, 114, 2, 70, "Subsection",
      CellTags->{"S1.12.1", "1.1"}],
    Cell[2089, 66, 488, 16, 70, "Text",
      CellTags->{"S1.12.1", "1.2"}],
    Cell[2580, 84, 295, 7, 70, "Text",
      CellTags->{"S1.12.1", "1.3"}],
    Cell[2878, 93, 341, 7, 70, "Text",
      CellTags->{"S1.12.1", "1.4"}],
    Cell[3222, 102, 664, 21, 70, "Text",
      CellTags->{"S1.12.1", "1.5"}],
    Cell[3889, 125, 429, 11, 70, "Text",
      CellTags->{"S1.12.1", "1.6"}],
    Cell[4321, 138, 429, 10, 70, "Text",
      CellTags->{"S1.12.1", "1.7"}],
    Cell[4753, 150, 272, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.8"}],
    Cell[5028, 161, 509, 11, 70, "Text",
      CellTags->{"S1.12.1", "1.9"}],
    Cell[5540, 174, 143, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.10"}],
    Cell[5708, 183, 90, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[5801, 187, 104, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[5920, 193, 204, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.11"}],
    Cell[6149, 202, 76, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[6228, 206, 107, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[6350, 212, 358, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.12"}],
    Cell[6711, 223, 375, 13, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.13"}],
    Cell[7111, 240, 101, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[7215, 244, 113, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[7343, 250, 223, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.14"}],
    Cell[7591, 259, 101, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[7695, 263, 113, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[7823, 269, 218, 5, 70, "Text",
      CellTags->{"S1.12.1", "1.15"}],
    Cell[8044, 276, 274, 6, 70, "Text",
      CellTags->{"S1.12.1", "1.16"}],
    Cell[8321, 284, 233, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.17"}],
    Cell[8579, 293, 96, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[8678, 297, 377, 6, 70, "Message",
      CellTags->"S1.12.1"],
    Cell[9058, 305, 249, 5, 70, "Message",
      CellTags->"S1.12.1"],
    Cell[9310, 312, 101, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[9426, 318, 564, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.18"}],
    Cell[9993, 329, 426, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.19"}],
    Cell[10422, 340, 270, 7, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.20"}],
    Cell[10717, 351, 74, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[10794, 355, 137, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[10946, 361, 332, 11, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.21"}],
    Cell[11303, 376, 82, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[11388, 380, 132, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[11535, 386, 261, 6, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.22"}],
    Cell[11821, 396, 76, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[11900, 400, 101, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[12016, 406, 546, 10, 70, "Text",
      CellTags->{"S1.12.1", "1.23"}],
    Cell[12565, 418, 202, 6, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.24"}],
    Cell[12792, 428, 76, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[12871, 432, 101, 3, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[12987, 438, 116, 2, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.25"}],
    Cell[13128, 444, 76, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[13207, 448, 76, 2, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[13298, 453, 158, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.26"}],
    Cell[13481, 462, 76, 2, 70, "Input",
      CellTags->"S1.12.1"],
    Cell[13560, 466, 75, 2, 70, "Output",
      CellTags->"S1.12.1"],
    Cell[13650, 471, 456, 8, 70, "Text",
      CellTags->{"S1.12.1", "1.27"}],
    Cell[14109, 481, 463, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.28"}],
    Cell[14575, 492, 331, 7, 70, "Text",
      CellTags->{"S1.12.1", "1.29"}]},
  "1.1"->{
    Cell[1972, 62, 114, 2, 70, "Subsection",
      CellTags->{"S1.12.1", "1.1"}]},
  "1.2"->{
    Cell[2089, 66, 488, 16, 70, "Text",
      CellTags->{"S1.12.1", "1.2"}]},
  "1.3"->{
    Cell[2580, 84, 295, 7, 70, "Text",
      CellTags->{"S1.12.1", "1.3"}]},
  "1.4"->{
    Cell[2878, 93, 341, 7, 70, "Text",
      CellTags->{"S1.12.1", "1.4"}]},
  "1.5"->{
    Cell[3222, 102, 664, 21, 70, "Text",
      CellTags->{"S1.12.1", "1.5"}]},
  "1.6"->{
    Cell[3889, 125, 429, 11, 70, "Text",
      CellTags->{"S1.12.1", "1.6"}]},
  "1.7"->{
    Cell[4321, 138, 429, 10, 70, "Text",
      CellTags->{"S1.12.1", "1.7"}]},
  "1.8"->{
    Cell[4753, 150, 272, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.8"}]},
  "1.9"->{
    Cell[5028, 161, 509, 11, 70, "Text",
      CellTags->{"S1.12.1", "1.9"}]},
  "1.10"->{
    Cell[5540, 174, 143, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.10"}]},
  "1.11"->{
    Cell[5920, 193, 204, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.11"}]},
  "1.12"->{
    Cell[6350, 212, 358, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.12"}]},
  "1.13"->{
    Cell[6711, 223, 375, 13, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.13"}]},
  "1.14"->{
    Cell[7343, 250, 223, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.14"}]},
  "1.15"->{
    Cell[7823, 269, 218, 5, 70, "Text",
      CellTags->{"S1.12.1", "1.15"}]},
  "1.16"->{
    Cell[8044, 276, 274, 6, 70, "Text",
      CellTags->{"S1.12.1", "1.16"}]},
  "1.17"->{
    Cell[8321, 284, 233, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.17"}]},
  "1.18"->{
    Cell[9426, 318, 564, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.18"}]},
  "1.19"->{
    Cell[9993, 329, 426, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.19"}]},
  "1.20"->{
    Cell[10422, 340, 270, 7, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.20"}]},
  "1.21"->{
    Cell[10946, 361, 332, 11, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.21"}]},
  "1.22"->{
    Cell[11535, 386, 261, 6, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.22"}]},
  "1.23"->{
    Cell[12016, 406, 546, 10, 70, "Text",
      CellTags->{"S1.12.1", "1.23"}]},
  "1.24"->{
    Cell[12565, 418, 202, 6, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.24"}]},
  "1.25"->{
    Cell[12987, 438, 116, 2, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.25"}]},
  "1.26"->{
    Cell[13298, 453, 158, 5, 70, "MathCaption",
      CellTags->{"S1.12.1", "1.26"}]},
  "1.27"->{
    Cell[13650, 471, 456, 8, 70, "Text",
      CellTags->{"S1.12.1", "1.27"}]},
  "1.28"->{
    Cell[14109, 481, 463, 9, 70, "Text",
      CellTags->{"S1.12.1", "1.28"}]},
  "1.29"->{
    Cell[14575, 492, 331, 7, 70, "Text",
      CellTags->{"S1.12.1", "1.29"}]},
  "S1.12.2"->{
    Cell[14943, 504, 88, 1, 70, "Subsection",
      CellTags->{"S1.12.2", "2.1"}],
    Cell[15034, 507, 579, 13, 70, "DefinitionBox",
      CellTags->"S1.12.2"],
    Cell[15616, 522, 132, 5, 70, "Caption",
      CellTags->"S1.12.2"],
    Cell[15751, 529, 237, 6, 70, "Text",
      CellTags->{"S1.12.2", "2.2"}],
    Cell[15991, 537, 528, 16, 70, "Text",
      CellTags->{"S1.12.2", "2.3"}],
    Cell[16522, 555, 530, 9, 70, "Text",
      CellTags->{"S1.12.2", "2.4"}],
    Cell[17055, 566, 515, 15, 70, "Text",
      CellTags->{"S1.12.2", "2.5"}],
    Cell[17573, 583, 264, 5, 70, "Text",
      CellTags->{"S1.12.2", "2.6"}],
    Cell[17840, 590, 326, 11, 70, "FramedBox",
      CellTags->"S1.12.2"],
    Cell[18169, 603, 140, 5, 70, "Caption",
      CellTags->"S1.12.2"],
    Cell[18312, 610, 404, 9, 70, "Text",
      CellTags->{"S1.12.2", "2.7"}],
    Cell[18719, 621, 316, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.8"}],
    Cell[19038, 631, 583, 21, 70, "FramedBox",
      CellTags->"S1.12.2"],
    Cell[19624, 654, 120, 5, 70, "Caption",
      CellTags->"S1.12.2"],
    Cell[19747, 661, 294, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.9"}],
    Cell[20044, 671, 428, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.10"}],
    Cell[20475, 681, 406, 11, 70, "FramedBox",
      CellTags->"S1.12.2"],
    Cell[20884, 694, 160, 5, 70, "Caption",
      CellTags->"S1.12.2"],
    Cell[21047, 701, 391, 9, 70, "Text",
      CellTags->{"S1.12.2", "2.11"}],
    Cell[21441, 712, 342, 7, 70, "Text",
      CellTags->{"S1.12.2", "2.12"}],
    Cell[21786, 721, 345, 7, 70, "Text",
      CellTags->{"S1.12.2", "2.13"}],
    Cell[22134, 730, 394, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.14"}],
    Cell[22531, 740, 256, 6, 70, "Text",
      CellTags->{"S1.12.2", "2.15"}],
    Cell[22790, 748, 345, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.16"}]},
  "2.1"->{
    Cell[14943, 504, 88, 1, 70, "Subsection",
      CellTags->{"S1.12.2", "2.1"}]},
  "2.2"->{
    Cell[15751, 529, 237, 6, 70, "Text",
      CellTags->{"S1.12.2", "2.2"}]},
  "2.3"->{
    Cell[15991, 537, 528, 16, 70, "Text",
      CellTags->{"S1.12.2", "2.3"}]},
  "2.4"->{
    Cell[16522, 555, 530, 9, 70, "Text",
      CellTags->{"S1.12.2", "2.4"}]},
  "2.5"->{
    Cell[17055, 566, 515, 15, 70, "Text",
      CellTags->{"S1.12.2", "2.5"}]},
  "2.6"->{
    Cell[17573, 583, 264, 5, 70, "Text",
      CellTags->{"S1.12.2", "2.6"}]},
  "2.7"->{
    Cell[18312, 610, 404, 9, 70, "Text",
      CellTags->{"S1.12.2", "2.7"}]},
  "2.8"->{
    Cell[18719, 621, 316, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.8"}]},
  "2.9"->{
    Cell[19747, 661, 294, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.9"}]},
  "2.10"->{
    Cell[20044, 671, 428, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.10"}]},
  "2.11"->{
    Cell[21047, 701, 391, 9, 70, "Text",
      CellTags->{"S1.12.2", "2.11"}]},
  "2.12"->{
    Cell[21441, 712, 342, 7, 70, "Text",
      CellTags->{"S1.12.2", "2.12"}]},
  "2.13"->{
    Cell[21786, 721, 345, 7, 70, "Text",
      CellTags->{"S1.12.2", "2.13"}]},
  "2.14"->{
    Cell[22134, 730, 394, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.14"}]},
  "2.15"->{
    Cell[22531, 740, 256, 6, 70, "Text",
      CellTags->{"S1.12.2", "2.15"}]},
  "2.16"->{
    Cell[22790, 748, 345, 8, 70, "Text",
      CellTags->{"S1.12.2", "2.16"}]},
  "S1.12.3"->{
    Cell[23172, 761, 129, 4, 70, "Subsection",
      CellTags->{"S1.12.3", "3.1"}],
    Cell[23304, 767, 680, 13, 70, "Text",
      CellTags->{"S1.12.3", "3.2"}],
    Cell[23987, 782, 235, 5, 70, "Text",
      CellTags->{"S1.12.3", "3.3"}],
    Cell[24225, 789, 388, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.4"}],
    Cell[24616, 799, 417, 7, 70, "Text",
      CellTags->{"S1.12.3", "3.5"}],
    Cell[25036, 808, 338, 7, 70, "Text",
      CellTags->{"S1.12.3", "3.6"}],
    Cell[25377, 817, 404, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.7"}],
    Cell[25784, 827, 719, 18, 70, "Text",
      CellTags->{"S1.12.3", "3.8"}],
    Cell[26506, 847, 377, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.9"}],
    Cell[26886, 857, 426, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.10"}],
    Cell[27315, 867, 288, 7, 70, "Text",
      CellTags->{"S1.12.3", "3.11"}],
    Cell[27606, 876, 311, 6, 70, "Text",
      CellTags->{"S1.12.3", "3.12"}],
    Cell[27920, 884, 350, 9, 70, "Text",
      CellTags->{"S1.12.3", "3.13"}]},
  "3.1"->{
    Cell[23172, 761, 129, 4, 70, "Subsection",
      CellTags->{"S1.12.3", "3.1"}]},
  "3.2"->{
    Cell[23304, 767, 680, 13, 70, "Text",
      CellTags->{"S1.12.3", "3.2"}]},
  "3.3"->{
    Cell[23987, 782, 235, 5, 70, "Text",
      CellTags->{"S1.12.3", "3.3"}]},
  "3.4"->{
    Cell[24225, 789, 388, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.4"}]},
  "3.5"->{
    Cell[24616, 799, 417, 7, 70, "Text",
      CellTags->{"S1.12.3", "3.5"}]},
  "3.6"->{
    Cell[25036, 808, 338, 7, 70, "Text",
      CellTags->{"S1.12.3", "3.6"}]},
  "3.7"->{
    Cell[25377, 817, 404, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.7"}]},
  "3.8"->{
    Cell[25784, 827, 719, 18, 70, "Text",
      CellTags->{"S1.12.3", "3.8"}]},
  "3.9"->{
    Cell[26506, 847, 377, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.9"}]},
  "3.10"->{
    Cell[26886, 857, 426, 8, 70, "Text",
      CellTags->{"S1.12.3", "3.10"}]},
  "3.11"->{
    Cell[27315, 867, 288, 7, 70, "Text",
      CellTags->{"S1.12.3", "3.11"}]},
  "3.12"->{
    Cell[27606, 876, 311, 6, 70, "Text",
      CellTags->{"S1.12.3", "3.12"}]},
  "3.13"->{
    Cell[27920, 884, 350, 9, 70, "Text",
      CellTags->{"S1.12.3", "3.13"}]},
  "S1.12.4"->{
    Cell[28307, 898, 139, 4, 70, "Subsection",
      CellTags->{"S1.12.4", "4.1"}],
    Cell[28449, 904, 487, 11, 70, "Text",
      CellTags->{"S1.12.4", "4.2"}],
    Cell[28939, 917, 390, 12, 70, "Text",
      CellTags->{"S1.12.4", "4.3"}],
    Cell[29332, 931, 305, 7, 70, "Text",
      CellTags->{"S1.12.4", "4.4"}],
    Cell[29640, 940, 312, 7, 70, "Text",
      CellTags->{"S1.12.4", "4.5"}],
    Cell[29955, 949, 221, 6, 70, "Text",
      CellTags->{"S1.12.4", "4.6"}],
    Cell[30179, 957, 481, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.7"}],
    Cell[30663, 967, 304, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.8"}],
    Cell[30970, 977, 311, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.9"}],
    Cell[31284, 987, 490, 9, 70, "Text",
      CellTags->{"S1.12.4", "4.10"}],
    Cell[31777, 998, 372, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.11"}],
    Cell[32152, 1008, 515, 10, 70, "Text",
      CellTags->{"S1.12.4", "4.12"}],
    Cell[32670, 1020, 405, 9, 70, "Text",
      CellTags->{"S1.12.4", "4.13"}]},
  "4.1"->{
    Cell[28307, 898, 139, 4, 70, "Subsection",
      CellTags->{"S1.12.4", "4.1"}]},
  "4.2"->{
    Cell[28449, 904, 487, 11, 70, "Text",
      CellTags->{"S1.12.4", "4.2"}]},
  "4.3"->{
    Cell[28939, 917, 390, 12, 70, "Text",
      CellTags->{"S1.12.4", "4.3"}]},
  "4.4"->{
    Cell[29332, 931, 305, 7, 70, "Text",
      CellTags->{"S1.12.4", "4.4"}]},
  "4.5"->{
    Cell[29640, 940, 312, 7, 70, "Text",
      CellTags->{"S1.12.4", "4.5"}]},
  "4.6"->{
    Cell[29955, 949, 221, 6, 70, "Text",
      CellTags->{"S1.12.4", "4.6"}]},
  "4.7"->{
    Cell[30179, 957, 481, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.7"}]},
  "4.8"->{
    Cell[30663, 967, 304, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.8"}]},
  "4.9"->{
    Cell[30970, 977, 311, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.9"}]},
  "4.10"->{
    Cell[31284, 987, 490, 9, 70, "Text",
      CellTags->{"S1.12.4", "4.10"}]},
  "4.11"->{
    Cell[31777, 998, 372, 8, 70, "Text",
      CellTags->{"S1.12.4", "4.11"}]},
  "4.12"->{
    Cell[32152, 1008, 515, 10, 70, "Text",
      CellTags->{"S1.12.4", "4.12"}]},
  "4.13"->{
    Cell[32670, 1020, 405, 9, 70, "Text",
      CellTags->{"S1.12.4", "4.13"}]},
  "S1.12.5"->{
    Cell[33112, 1034, 85, 1, 70, "Subsection",
      CellTags->{"S1.12.5", "5.1"}],
    Cell[33200, 1037, 313, 8, 70, "Text",
      CellTags->{"S1.12.5", "5.2"}],
    Cell[33516, 1047, 444, 8, 72, "Text",
      CellTags->{"S1.12.5", "5.3"}],
    Cell[33963, 1057, 133, 4, 70, "Text",
      CellTags->{"S1.12.5", "5.4"}],
    Cell[34099, 1063, 929, 22, 188, "Notes",
      CellTags->{"S1.12.5", "5.5"}],
    Cell[35031, 1087, 213, 5, 40, "Text",
      CellTags->{"S1.12.5", "5.6"}],
    Cell[35247, 1094, 442, 10, 72, "Text",
      CellTags->{"S1.12.5", "5.7"}],
    Cell[35692, 1106, 385, 10, 56, "Text",
      CellTags->{"S1.12.5", "5.8"}],
    Cell[36080, 1118, 348, 9, 56, "Text",
      CellTags->{"S1.12.5", "5.9"}],
    Cell[36431, 1129, 222, 5, 40, "Text",
      CellTags->{"S1.12.5", "5.10"}],
    Cell[36656, 1136, 212, 6, 40, "Text",
      CellTags->{"S1.12.5", "5.11"}],
    Cell[36871, 1144, 463, 10, 88, "Text",
      CellTags->{"S1.12.5", "5.12"}],
    Cell[37337, 1156, 287, 6, 56, "Text",
      CellTags->{"S1.12.5", "5.13"}],
    Cell[37627, 1164, 484, 12, 72, "Text",
      CellTags->{"S1.12.5", "5.14"}],
    Cell[38114, 1178, 280, 6, 56, "Text",
      CellTags->{"S1.12.5", "5.15"}]},
  "5.1"->{
    Cell[33112, 1034, 85, 1, 70, "Subsection",
      CellTags->{"S1.12.5", "5.1"}]},
  "5.2"->{
    Cell[33200, 1037, 313, 8, 70, "Text",
      CellTags->{"S1.12.5", "5.2"}]},
  "5.3"->{
    Cell[33516, 1047, 444, 8, 72, "Text",
      CellTags->{"S1.12.5", "5.3"}]},
  "5.4"->{
    Cell[33963, 1057, 133, 4, 70, "Text",
      CellTags->{"S1.12.5", "5.4"}]},
  "5.5"->{
    Cell[34099, 1063, 929, 22, 188, "Notes",
      CellTags->{"S1.12.5", "5.5"}]},
  "5.6"->{
    Cell[35031, 1087, 213, 5, 40, "Text",
      CellTags->{"S1.12.5", "5.6"}]},
  "5.7"->{
    Cell[35247, 1094, 442, 10, 72, "Text",
      CellTags->{"S1.12.5", "5.7"}]},
  "5.8"->{
    Cell[35692, 1106, 385, 10, 56, "Text",
      CellTags->{"S1.12.5", "5.8"}]},
  "5.9"->{
    Cell[36080, 1118, 348, 9, 56, "Text",
      CellTags->{"S1.12.5", "5.9"}]},
  "5.10"->{
    Cell[36431, 1129, 222, 5, 40, "Text",
      CellTags->{"S1.12.5", "5.10"}]},
  "5.11"->{
    Cell[36656, 1136, 212, 6, 40, "Text",
      CellTags->{"S1.12.5", "5.11"}]},
  "5.12"->{
    Cell[36871, 1144, 463, 10, 88, "Text",
      CellTags->{"S1.12.5", "5.12"}]},
  "5.13"->{
    Cell[37337, 1156, 287, 6, 56, "Text",
      CellTags->{"S1.12.5", "5.13"}]},
  "5.14"->{
    Cell[37627, 1164, 484, 12, 72, "Text",
      CellTags->{"S1.12.5", "5.14"}]},
  "5.15"->{
    Cell[38114, 1178, 280, 6, 56, "Text",
      CellTags->{"S1.12.5", "5.15"}]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"S1.12.0", 39038, 1204},
  {"0.1", 39129, 1207},
  {"S1.12.1", 39224, 1210},
  {"1.1", 43258, 1317},
  {"1.2", 43352, 1320},
  {"1.3", 43441, 1323},
  {"1.4", 43529, 1326},
  {"1.5", 43617, 1329},
  {"1.6", 43707, 1332},
  {"1.7", 43797, 1335},
  {"1.8", 43887, 1338},
  {"1.9", 43976, 1341},
  {"1.10", 44067, 1344},
  {"1.11", 44165, 1347},
  {"1.12", 44263, 1350},
  {"1.13", 44354, 1353},
  {"1.14", 44453, 1356},
  {"1.15", 44551, 1359},
  {"1.16", 44642, 1362},
  {"1.17", 44733, 1365},
  {"1.18", 44831, 1368},
  {"1.19", 44922, 1371},
  {"1.20", 45013, 1374},
  {"1.21", 45112, 1377},
  {"1.22", 45212, 1380},
  {"1.23", 45311, 1383},
  {"1.24", 45404, 1386},
  {"1.25", 45503, 1389},
  {"1.26", 45602, 1392},
  {"1.27", 45701, 1395},
  {"1.28", 45793, 1398},
  {"1.29", 45885, 1401},
  {"S1.12.2", 45980, 1404},
  {"2.1", 47846, 1453},
  {"2.2", 47941, 1456},
  {"2.3", 48031, 1459},
  {"2.4", 48122, 1462},
  {"2.5", 48212, 1465},
  {"2.6", 48303, 1468},
  {"2.7", 48393, 1471},
  {"2.8", 48483, 1474},
  {"2.9", 48573, 1477},
  {"2.10", 48664, 1480},
  {"2.11", 48756, 1483},
  {"2.12", 48848, 1486},
  {"2.13", 48940, 1489},
  {"2.14", 49032, 1492},
  {"2.15", 49124, 1495},
  {"2.16", 49216, 1498},
  {"S1.12.3", 49311, 1501},
  {"3.1", 50349, 1528},
  {"3.2", 50445, 1531},
  {"3.3", 50536, 1534},
  {"3.4", 50626, 1537},
  {"3.5", 50716, 1540},
  {"3.6", 50806, 1543},
  {"3.7", 50896, 1546},
  {"3.8", 50986, 1549},
  {"3.9", 51077, 1552},
  {"3.10", 51168, 1555},
  {"3.11", 51260, 1558},
  {"3.12", 51352, 1561},
  {"3.13", 51444, 1564},
  {"S1.12.4", 51539, 1567},
  {"4.1", 52580, 1594},
  {"4.2", 52676, 1597},
  {"4.3", 52767, 1600},
  {"4.4", 52858, 1603},
  {"4.5", 52948, 1606},
  {"4.6", 53038, 1609},
  {"4.7", 53128, 1612},
  {"4.8", 53218, 1615},
  {"4.9", 53308, 1618},
  {"4.10", 53399, 1621},
  {"4.11", 53491, 1624},
  {"4.12", 53583, 1627},
  {"4.13", 53677, 1630},
  {"S1.12.5", 53773, 1633},
  {"5.1", 54988, 1664},
  {"5.2", 55084, 1667},
  {"5.3", 55175, 1670},
  {"5.4", 55266, 1673},
  {"5.5", 55357, 1676},
  {"5.6", 55451, 1679},
  {"5.7", 55542, 1682},
  {"5.8", 55634, 1685},
  {"5.9", 55726, 1688},
  {"5.10", 55818, 1691},
  {"5.11", 55911, 1694},
  {"5.12", 56004, 1697},
  {"5.13", 56098, 1700},
  {"5.14", 56191, 1703},
  {"5.15", 56285, 1706}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1776, 53, 171, 5, 91, "Section",
  CellTags->{"S1.12.0", "0.1"}],

Cell[CellGroupData[{
Cell[1972, 62, 114, 2, 70, "Subsection",
  CellTags->{"S1.12.1", "1.1"}],
Cell[2089, 66, 488, 16, 70, "Text",
  CellTags->{"S1.12.1", "1.2"}],
Cell[2580, 84, 295, 7, 70, "Text",
  CellTags->{"S1.12.1", "1.3"}],
Cell[2878, 93, 341, 7, 70, "Text",
  CellTags->{"S1.12.1", "1.4"}],
Cell[3222, 102, 664, 21, 70, "Text",
  CellTags->{"S1.12.1", "1.5"}],
Cell[3889, 125, 429, 11, 70, "Text",
  CellTags->{"S1.12.1", "1.6"}],
Cell[4321, 138, 429, 10, 70, "Text",
  CellTags->{"S1.12.1", "1.7"}],
Cell[4753, 150, 272, 9, 70, "Text",
  CellTags->{"S1.12.1", "1.8"}],
Cell[5028, 161, 509, 11, 70, "Text",
  CellTags->{"S1.12.1", "1.9"}],
Cell[5540, 174, 143, 5, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.10"}],

Cell[CellGroupData[{
Cell[5708, 183, 90, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[5801, 187, 104, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[5920, 193, 204, 5, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.11"}],

Cell[CellGroupData[{
Cell[6149, 202, 76, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[6228, 206, 107, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[6350, 212, 358, 9, 70, "Text",
  CellTags->{"S1.12.1", "1.12"}],
Cell[6711, 223, 375, 13, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.13"}],

Cell[CellGroupData[{
Cell[7111, 240, 101, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[7215, 244, 113, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[7343, 250, 223, 5, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.14"}],

Cell[CellGroupData[{
Cell[7591, 259, 101, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[7695, 263, 113, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[7823, 269, 218, 5, 70, "Text",
  CellTags->{"S1.12.1", "1.15"}],
Cell[8044, 276, 274, 6, 70, "Text",
  CellTags->{"S1.12.1", "1.16"}],
Cell[8321, 284, 233, 5, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.17"}],

Cell[CellGroupData[{
Cell[8579, 293, 96, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[8678, 297, 377, 6, 70, "Message",
  CellTags->"S1.12.1"],
Cell[9058, 305, 249, 5, 70, "Message",
  CellTags->"S1.12.1"],
Cell[9310, 312, 101, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[9426, 318, 564, 9, 70, "Text",
  CellTags->{"S1.12.1", "1.18"}],
Cell[9993, 329, 426, 9, 70, "Text",
  CellTags->{"S1.12.1", "1.19"}],
Cell[10422, 340, 270, 7, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.20"}],

Cell[CellGroupData[{
Cell[10717, 351, 74, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[10794, 355, 137, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[10946, 361, 332, 11, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.21"}],

Cell[CellGroupData[{
Cell[11303, 376, 82, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[11388, 380, 132, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[11535, 386, 261, 6, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.22"}],

Cell[CellGroupData[{
Cell[11821, 396, 76, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[11900, 400, 101, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[12016, 406, 546, 10, 70, "Text",
  CellTags->{"S1.12.1", "1.23"}],
Cell[12565, 418, 202, 6, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.24"}],

Cell[CellGroupData[{
Cell[12792, 428, 76, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[12871, 432, 101, 3, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[12987, 438, 116, 2, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.25"}],

Cell[CellGroupData[{
Cell[13128, 444, 76, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[13207, 448, 76, 2, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[13298, 453, 158, 5, 70, "MathCaption",
  CellTags->{"S1.12.1", "1.26"}],

Cell[CellGroupData[{
Cell[13481, 462, 76, 2, 70, "Input",
  CellTags->"S1.12.1"],
Cell[13560, 466, 75, 2, 70, "Output",
  CellTags->"S1.12.1"]
}, Open  ]],
Cell[13650, 471, 456, 8, 70, "Text",
  CellTags->{"S1.12.1", "1.27"}],
Cell[14109, 481, 463, 9, 70, "Text",
  CellTags->{"S1.12.1", "1.28"}],
Cell[14575, 492, 331, 7, 70, "Text",
  CellTags->{"S1.12.1", "1.29"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[14943, 504, 88, 1, 70, "Subsection",
  CellTags->{"S1.12.2", "2.1"}],
Cell[15034, 507, 579, 13, 70, "DefinitionBox",
  CellTags->"S1.12.2"],
Cell[15616, 522, 132, 5, 70, "Caption",
  CellTags->"S1.12.2"],
Cell[15751, 529, 237, 6, 70, "Text",
  CellTags->{"S1.12.2", "2.2"}],
Cell[15991, 537, 528, 16, 70, "Text",
  CellTags->{"S1.12.2", "2.3"}],
Cell[16522, 555, 530, 9, 70, "Text",
  CellTags->{"S1.12.2", "2.4"}],
Cell[17055, 566, 515, 15, 70, "Text",
  CellTags->{"S1.12.2", "2.5"}],
Cell[17573, 583, 264, 5, 70, "Text",
  CellTags->{"S1.12.2", "2.6"}],
Cell[17840, 590, 326, 11, 70, "FramedBox",
  CellTags->"S1.12.2"],
Cell[18169, 603, 140, 5, 70, "Caption",
  CellTags->"S1.12.2"],
Cell[18312, 610, 404, 9, 70, "Text",
  CellTags->{"S1.12.2", "2.7"}],
Cell[18719, 621, 316, 8, 70, "Text",
  CellTags->{"S1.12.2", "2.8"}],
Cell[19038, 631, 583, 21, 70, "FramedBox",
  CellTags->"S1.12.2"],
Cell[19624, 654, 120, 5, 70, "Caption",
  CellTags->"S1.12.2"],
Cell[19747, 661, 294, 8, 70, "Text",
  CellTags->{"S1.12.2", "2.9"}],
Cell[20044, 671, 428, 8, 70, "Text",
  CellTags->{"S1.12.2", "2.10"}],
Cell[20475, 681, 406, 11, 70, "FramedBox",
  CellTags->"S1.12.2"],
Cell[20884, 694, 160, 5, 70, "Caption",
  CellTags->"S1.12.2"],
Cell[21047, 701, 391, 9, 70, "Text",
  CellTags->{"S1.12.2", "2.11"}],
Cell[21441, 712, 342, 7, 70, "Text",
  CellTags->{"S1.12.2", "2.12"}],
Cell[21786, 721, 345, 7, 70, "Text",
  CellTags->{"S1.12.2", "2.13"}],
Cell[22134, 730, 394, 8, 70, "Text",
  CellTags->{"S1.12.2", "2.14"}],
Cell[22531, 740, 256, 6, 70, "Text",
  CellTags->{"S1.12.2", "2.15"}],
Cell[22790, 748, 345, 8, 70, "Text",
  CellTags->{"S1.12.2", "2.16"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[23172, 761, 129, 4, 70, "Subsection",
  CellTags->{"S1.12.3", "3.1"}],
Cell[23304, 767, 680, 13, 70, "Text",
  CellTags->{"S1.12.3", "3.2"}],
Cell[23987, 782, 235, 5, 70, "Text",
  CellTags->{"S1.12.3", "3.3"}],
Cell[24225, 789, 388, 8, 70, "Text",
  CellTags->{"S1.12.3", "3.4"}],
Cell[24616, 799, 417, 7, 70, "Text",
  CellTags->{"S1.12.3", "3.5"}],
Cell[25036, 808, 338, 7, 70, "Text",
  CellTags->{"S1.12.3", "3.6"}],
Cell[25377, 817, 404, 8, 70, "Text",
  CellTags->{"S1.12.3", "3.7"}],
Cell[25784, 827, 719, 18, 70, "Text",
  CellTags->{"S1.12.3", "3.8"}],
Cell[26506, 847, 377, 8, 70, "Text",
  CellTags->{"S1.12.3", "3.9"}],
Cell[26886, 857, 426, 8, 70, "Text",
  CellTags->{"S1.12.3", "3.10"}],
Cell[27315, 867, 288, 7, 70, "Text",
  CellTags->{"S1.12.3", "3.11"}],
Cell[27606, 876, 311, 6, 70, "Text",
  CellTags->{"S1.12.3", "3.12"}],
Cell[27920, 884, 350, 9, 70, "Text",
  CellTags->{"S1.12.3", "3.13"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[28307, 898, 139, 4, 70, "Subsection",
  CellTags->{"S1.12.4", "4.1"}],
Cell[28449, 904, 487, 11, 70, "Text",
  CellTags->{"S1.12.4", "4.2"}],
Cell[28939, 917, 390, 12, 70, "Text",
  CellTags->{"S1.12.4", "4.3"}],
Cell[29332, 931, 305, 7, 70, "Text",
  CellTags->{"S1.12.4", "4.4"}],
Cell[29640, 940, 312, 7, 70, "Text",
  CellTags->{"S1.12.4", "4.5"}],
Cell[29955, 949, 221, 6, 70, "Text",
  CellTags->{"S1.12.4", "4.6"}],
Cell[30179, 957, 481, 8, 70, "Text",
  CellTags->{"S1.12.4", "4.7"}],
Cell[30663, 967, 304, 8, 70, "Text",
  CellTags->{"S1.12.4", "4.8"}],
Cell[30970, 977, 311, 8, 70, "Text",
  CellTags->{"S1.12.4", "4.9"}],
Cell[31284, 987, 490, 9, 70, "Text",
  CellTags->{"S1.12.4", "4.10"}],
Cell[31777, 998, 372, 8, 70, "Text",
  CellTags->{"S1.12.4", "4.11"}],
Cell[32152, 1008, 515, 10, 70, "Text",
  CellTags->{"S1.12.4", "4.12"}],
Cell[32670, 1020, 405, 9, 70, "Text",
  CellTags->{"S1.12.4", "4.13"}]
}, Open  ]],

Cell[CellGroupData[{
Cell[33112, 1034, 85, 1, 70, "Subsection",
  CellTags->{"S1.12.5", "5.1"}],
Cell[33200, 1037, 313, 8, 70, "Text",
  CellTags->{"S1.12.5", "5.2"}],
Cell[33516, 1047, 444, 8, 72, "Text",
  CellTags->{"S1.12.5", "5.3"}],
Cell[33963, 1057, 133, 4, 70, "Text",
  CellTags->{"S1.12.5", "5.4"}],
Cell[34099, 1063, 929, 22, 188, "Notes",
  CellTags->{"S1.12.5", "5.5"}],
Cell[35031, 1087, 213, 5, 40, "Text",
  CellTags->{"S1.12.5", "5.6"}],
Cell[35247, 1094, 442, 10, 72, "Text",
  CellTags->{"S1.12.5", "5.7"}],
Cell[35692, 1106, 385, 10, 56, "Text",
  CellTags->{"S1.12.5", "5.8"}],
Cell[36080, 1118, 348, 9, 56, "Text",
  CellTags->{"S1.12.5", "5.9"}],
Cell[36431, 1129, 222, 5, 40, "Text",
  CellTags->{"S1.12.5", "5.10"}],
Cell[36656, 1136, 212, 6, 40, "Text",
  CellTags->{"S1.12.5", "5.11"}],
Cell[36871, 1144, 463, 10, 88, "Text",
  CellTags->{"S1.12.5", "5.12"}],
Cell[37337, 1156, 287, 6, 56, "Text",
  CellTags->{"S1.12.5", "5.13"}],
Cell[37627, 1164, 484, 12, 72, "Text",
  CellTags->{"S1.12.5", "5.14"}],
Cell[38114, 1178, 280, 6, 56, "Text",
  CellTags->{"S1.12.5", "5.15"}]
}, Open  ]]
}, Open  ]]
}
]
*)



(*******************************************************************
End of Mathematica Notebook file.
*******************************************************************)

