(* Author:          Christopher Williamson *)
(* Copyright:       Copyright 2004-2005, Wolfram Research, Inc. *)

DatabaseResourcesPath::install = "Multiple installations of Database Package exist \
at `1`. This may lead to unpredictable results when running Database Package.";

Begin[ "`DataSources`Private`" ] 

(* Data Path *)
addedResourceDirectories = {}

AddDatabaseResources[ x_String] :=
  addedResourceDirectories = Join[ {x}, addedResourceDirectories]

autoDatabaseResourcesPath[] :=
  Module[{dir, appPaths = {}, appDirs, dataDirs},
    dir = If[ NameQ["$InstallationDirectory"], ToExpression["$InstallationDirectory"], $TopDirectory];
    If[StringQ[dir],
      PrependTo[appPaths, ToFileName[{dir, "AddOns", "ExtraPackages"}]];
      PrependTo[appPaths, ToFileName[{dir, "AddOns", "StandardPackages"}]];
      PrependTo[appPaths, ToFileName[{dir, "AddOns", "Autoload"}]];
      PrependTo[appPaths, ToFileName[{dir, "AddOns", "Applications"}]];
      ];
    dir = If[ NameQ["$BaseDirectory"], ToExpression["$BaseDirectory"], $AddOnsDirectory];
    If[StringQ[dir],
      (* This branch is for 4.2 and later (4.1.5 on Mac OSX). *)
      PrependTo[appPaths, ToFileName[{dir, "Autoload"}]];
      PrependTo[appPaths, ToFileName[{dir, "Applications"}]];
      ];
    dir = If[ NameQ["$UserBaseDirectory"], ToExpression["$UserBaseDirectory"], $UserAddOnsDirectory];
    If[StringQ[dir],
      (* 4.2 and later *)
      PrependTo[appPaths, ToFileName[{dir, "Autoload"}]];
      PrependTo[appPaths, ToFileName[{dir, "Applications"}]];
      ,
      (* else *)
      PrependTo[appPaths, ToFileName[{$PreferencesDirectory, "AddOns", "Autoload"}]];
      PrependTo[appPaths, ToFileName[{$PreferencesDirectory, "AddOns", "Applications"}]];
      ];

    (* Add extension directories. *)
  If[ MatchQ[ addedResourceDirectories, {_String..}],
    appPaths = Join[ addedResourceDirectories, appPaths]];
     
    (* FileNames sorts all results so we need to apply this to each appPaths separate to 
       preserve desired path order *)
    appDirs = Select[ Flatten[FileNames["*", #]& /@ appPaths], (FileType[#] === Directory)&];
  
    dataDirs = Select[ToFileName[{#, "DatabaseResources"}]& /@ appDirs, (FileType[#] === Directory)&];
    
    (* Here we put the special WRI SystemFiles dir first. The thinking for putting it absolutely first
        in the search path is that it becomes a convenient place to put Data and be sure they override any
        provided by applications. It can be used to resolve application conflicts in this way.
     *)
    PrependTo[dataDirs, ToFileName[{$BaseDirectory, "DatabaseResources"}]];
    PrependTo[dataDirs, ToFileName[{$UserBaseDirectory, "DatabaseResources"}]];    
     
    dataDirs
    ];

DatabaseResourcesPath[] := autoDatabaseResourcesPath[];

(* Returns True if the pathname begins with a
   relative path metacharacter *)
(* for MacOS *)
beginsRelativeMetaCharQ[str_String] :=
   (StringMatchQ[str, ":"] || StringMatchQ[str, "::"] ||
    StringMatchQ[str, ToFileName[{":"},"*"]] ||
    StringMatchQ[str, ToFileName[{"::"},"*"]]) /;
      StringMatchQ[$OperatingSystem, "MacOS"]
(* for non-MacOS *)
beginsRelativeMetaCharQ[str_String] :=
   StringMatchQ[str, "."] || StringMatchQ[str, ".."] ||
   StringMatchQ[str, ToFileName[{"."},"*"]] ||
   StringMatchQ[str, ToFileName[{".."},"*"]]
beginsRelativeMetaCharQ[___] := False

findDataFile[file_String] :=
  Block[{ffile},
    ffile = System`Private`FindFile[file];
    If[ StringQ[ffile],
      ffile = System`Private`ExpandFileName[ffile];
      If[ beginsRelativeMetaCharQ[ffile],
        ffile = ToFileName[{Directory[]}, ffile] ];
      If[ FileType[ffile] =!= File, ffile = $Failed];
    (* else *),
    ffile = $Failed];
  ffile
  ]
  
resolveMathematicaFile[file_String, otherDirs___] :=
  Block[{$Path = Join[ {otherDirs}, DatabaseResourcesPath[], $Path], 
         ffile = $Failed},
    ffile = findDataFile[file <> ".m"]
  ]

dataSourceQ[file_String] := 
  Module[{is, word},
    is = OpenRead[file];
    word = Read[is, Word , WordSeparators -> {" ", "\n", "\r", "\t", "["}];
    Close[is];
    word === "SQLConnection"
  ]

autoDataSources[] := 
  Module[{ms, dss = {}},
    ms = Select[ Flatten[FileNames["*.m", #]& /@ DatabaseResourcesPath[]], (FileType[#] =!= Directory)&];
    
    If[dataSourceQ[#],
      AppendTo[dss, Append[Get[#], "Location" -> #]]
    ] & /@ ms;
    dss
  ]
  
DataSources[] := autoDataSources[];

DataSources[dataSourceName_String]:=
  Module[{cases},
    cases = Cases[DataSources[], SQLConnection[___, "Name"->dataSourceName,___]];
    If[cases === {}, Null, First[cases]]
  ];

DataSourceNames[] := 
  ("Name" /. canonicalOptions[Options[#]] /. Options[SQLConnection]) & /@ DataSources[];

queryQ[file_String] := 
  Module[{is, word},
    is = OpenRead[file];
    word = Read[is, Word , WordSeparators -> {" ", "\n", "\r", "\t", "["}];
    Close[is];
    word === "SQLSelect"
  ]

autoQueries[] := 
  Module[{ms, qs = {}},
    ms = Select[ Flatten[FileNames["*.m", #]& /@ DatabaseResourcesPath[]], (FileType[#] =!= Directory)&];
    
    If[queryQ[#],
      AppendTo[qs, Append[Get[#], "Location" -> #]]
    ] & /@ ms;
    qs
  ]
  
SQLQueries[] := autoQueries[];

SQLQueryNames[] := 
  ("Name" /. canonicalOptions[Options[#]] /. Options[SQLSelect]) & /@ SQLQueries[];

End[]
