(* Author:          Christopher Williamson *)
(* Copyright:       Copyright 2004-2005, Wolfram Research, Inc. *)

Begin[ "`SQL`Private`"] 

Needs["JLink`"];

(* Initialized variables *)

$SQLTimeout = None;

$sqlConnections = {};

SQLConnections[] := $sqlConnections;

$resultSets = {};

SQLResultSets[] := $resultSets;

(* ERROR MESSAGES *)

JDBC::error = 
"JDBC error: `1`"

OpenSQLConnection::notfound = 
"DataSource not found: `1`"

OpenSQLConnection::nogui = 
"GUIKit` is required for this functionality.  Please go to \
http://www.wolfram.com to download this package."

OpenSQLConnection::timeout = 
"Illegal value for Timeout option: `1`"

SQLBeginTransaction::nested = 
"Nested transactions are not allowed.  Continuing with the first \
transaction."

SQLColumn::columnname = 
"The ColumnName was not of the proper type or was not provided."

SQLColumn::datalength = 
"DataLength must be specified for creation and must be an Integer \
value."

SQLColumn::datatype = 
"DataTypeName must be specified for creation and must be a String value."

SQLColumns::opt = 
"Invalid value for option: `1`"

SQLConnection::conn = 
"Connection is not opened properly."

DatabaseExplorer::nogui = 
"GUIKit` is required for this functionality.  Please go to \
http://www.wolfram.com to download this package."

SQLExecute::maxrows = 
"Illegal value for MaxRows option: `1`"

SQLExecute::timeout = 
"Illegal value for Timeout option: `1`"

SQLResultSet::forwardonly = 
"This SQLResultSet is required to be ForwardOnly."

SQLResultSetOpen::mode = 
"Invalid SQLResultSet mode: `1`"

SQLResultSetTake::invalidrange = 
"Invalid range: `1`"

SQLSetSavepoint::version = 
"This feature requires Java 1.4."

SQLTables::opt = 
"Invalid value for option: `1`"

SQLValue::illegalvalue = 
"The value `1` cannot be converted to a value in an SQL statement."

ThrowException[symbol_Symbol, tagname_String, message_String] := 
  Module[{msg = GetJavaException[]@getMessage[]}, 
    Switch[
      msg, 
      Null,       
      JLink`Exceptions`Private`$internalJavaExceptionHandler[symbol, tagname, message],
      _,
      Message[JDBC::error, msg]
    ];
    Throw[$Failed];
  ];

(* Options *)

Options[ JDBCDriver ] := 
    {
      "Name" -> "" , 
      "Description" -> "" , 
      "Driver" -> "", 
      "Protocol" -> "",
      "Location" -> ""
    }

Options[ SQLConnection ] := 
    {
      "Name" -> "", 
      "Description" -> "", 
      "Username" -> "", 
      "Password" -> "",
      "Location" -> "",
      "RelativePath" -> False,
      "Version"->""
    }

Options[ OpenSQLConnection ] = 
    Union[
      Options[SQLConnection],
      {
        "Timeout"->$SQLTimeout
      }
    ]

Options[ SQLResultSetOpen ] = 
    {
      "Mode"->"ScrollInsensitive"
    }

Options[ SQLColumn ] = 
    { 
      "DataTypeName" -> None, 
      "DataLength" -> None,
      "Nullable" -> None
    }

Options[ SQLColumnInformation] = 
    { 
      "Catalog" -> None, 
      "Schema" -> None, 
      "ShowColumnHeadings"->False
    }

Options[ SQLColumnNames ] = 
    { 
      "Catalog" -> None, 
      "Schema" -> None
    }

Options[ SQLColumns ] = 
    { 
      "Catalog" -> None, 
      "Schema" -> None
    }

Options[SQLCreateTable] := 
    {
      "Timeout"->$SQLTimeout
    }

Options[ SQLDataTypeInformation ] = 
    { 
      "ShowColumnHeadings"->False
    }

Options[SQLDelete] := 
    {
      "Timeout"->$SQLTimeout
    }

Options[SQLDropTable] := 
    {
      "Timeout"->$SQLTimeout
    }

Options[ SQLExecute ] = 
    { 
      "GetAsStrings" -> False, 
      "MaxRows" -> Automatic, 
      "ShowColumnHeadings" -> False,
      "Timeout"->$SQLTimeout
    }

Options[SQLInsert] := 
    {
      "Timeout"->$SQLTimeout
    }

Options[ SQLResultSetRead ] = 
    { 
      "GetAsStrings" -> False
    }

Options[ SQLResultSetCurrent ] = 
    { 
      "GetAsStrings" -> False
    }

Options[ SQLResultSetTake ] = 
    { 
      "GetAsStrings" -> False
    }

Options[ SQLSelect ] = 
    Join[
      { 
        "SortingColumns" -> None, 
        "Distinct"->False 
      },
      Options[SQLExecute]
    ]

Options[ SQLTable ] = 
    { 
      "TableType" -> $DefaultTableType
    }

Options[ SQLTableInformation ] = 
    { 
      "Catalog" -> None,
      "Schema" -> None,
      "ShowColumnHeadings"->False,
      "TableType" -> $DefaultTableType
    }

Options[ SQLTableNames ] = 
    { 
      "Catalog" -> None,
      "Schema" -> None,
      "TableType" -> $DefaultTableType 
    }

Options[ SQLTables ] = 
    { 
      "Catalog" -> None,
      "Schema" -> None,
      "TableType" -> $DefaultTableType 
    }

Options[SQLUpdate] := 
    {
      "Timeout"->$SQLTimeout
    }

Options[SQLSavepoint] :=
    {
      "Name"->""
    }

  
jdbcDriverQ[file_String] := 
  Module[{is, word},
    is = OpenRead[file];
    word = Read[is, Word , WordSeparators -> {" ", "\n", "\r", "\t", "["}];
    Close[is];
    word === "JDBCDriver"
  ]

autoJDBCDrivers[] := 
  Module[{ms, drivers = {}},
    ms = Select[ Flatten[FileNames["*.m", #]& /@ DatabaseResourcesPath[]], (FileType[#] =!= Directory)&];
    
    If[jdbcDriverQ[#],
      AppendTo[drivers, Append[Get[#], "Location" -> #]]
    ] & /@ ms;
    drivers
  ]

JDBCDrivers[] := autoJDBCDrivers[];

JDBCDrivers[driverName_String]:=
  Module[{cases},
    cases = Cases[JDBCDrivers[], JDBCDriver[___, "Name"->driverName,___]];
    If[cases === {}, Null, First[cases]]
  ];
    
JDBCDriverNames[] := 
  ("Name" /. canonicalOptions[Options[#]] /. Options[ JDBCDriver]) & /@ JDBCDrivers[];

(* SQLConnection Functionality *)

connectionIndex = 0;

OpenSQLConnection[opts___Rule] := 
  Module[{load},
    load = 
      Check[
        Needs["GUIKit`"], 
        Message[OpenSQLConnection::nogui]; $Failed, 
        Get::noopen, Needs::nocont];
    If[load === $Failed, Return[$Failed]];
    GUIKit`GUIRunModal["DatabaseLink/DataSourceFrame"]
  ]

OpenSQLConnection[ JDBC[
                          driver_JDBCDriver, 
                          url_
                        ], 
                   opts___Rule] := 
  Module[{name}, 
    name = "Name" /. canonicalOptions[Options[driver]] /. Options[ JDBCDriver ];
    OpenSQLConnection[ JDBC[name, url], opts]
  ]
  
processUrlProperty[str_String] :=
  Module[{one, two, sym},
    If[StringMatchQ[str, "*${*}*"], 
      one = Last[Flatten[StringPosition[str,"${",1]]]+1;
      two = First[Flatten[StringPosition[StringTake[str, {one, StringLength[str]}], "}"]]] - 2;
      sym = StringTake[str, {one, two+one}];
      StringJoin[
        {
          StringTake[str, {1, one - 3}], 
          ToString[ToExpression[sym]], 
          StringTake[str, {one + two + 2, StringLength[str]}]
        }
      ],
      str
    ]
  ]

processUrlProperty[prop_] := prop;
  
OpenSQLConnection[ JDBC[
                          driver_String, 
                          url_String
                        ], 
                   opts___Rule] := 
  JavaBlock[
    Module[ {id, connection, conn, d = driver, jdbc, u = url, p, useOpts, user, 
             password, loc, rel, load, timeout}, 
      Block[{$JavaExceptionHandler = ThrowException},
        Catch[
          useOpts = canonicalOptions[Flatten[{opts}]];
          loc      = "Location" /. useOpts /. Options[ OpenSQLConnection ];
          rel      = "RelativePath" /. useOpts /. Options[ OpenSQLConnection ];
          user     = "Username" /. useOpts /. Options[ OpenSQLConnection ]; 
          password = "Password" /. useOpts /. Options[ OpenSQLConnection ]; 
          timeout  = "Timeout" /. useOpts /. Options[OpenSQLConnection];
    
          If[TrueQ[rel],
            dir = DirectoryName[loc];
            u = dir <> u;
          ];
      
          If[MemberQ[JDBCDriverNames[], driver], 
            pos = First[Flatten[Position[JDBCDriverNames[], driver]]];
            jdbc = JDBCDrivers[][[pos]];
            {d, p} = {"Driver", "Protocol"} /. canonicalOptions[Options[jdbc]] /. Options[ JDBCDriver ];
            If[p =!= "Protocol" && !StringMatchQ[u, p <> "*"],
              u = p <> u;
            ]
          ];

          InstallJava[];
          LoadClass["com.wolfram.jlink.JLinkClassLoader"];
          JLinkClassLoader`classFromName[d]@newInstance[];
          LoadClass["com.wolfram.databaselink.JDBCConnectionManager"];
        
          If[StringMatchQ[password, "$Prompt"], 
            load = 
              Check[
                Needs["GUIKit`"], 
                Message[OpenSQLConnection::nogui]; $Failed, 
                Get::noopen, Needs::nocont];
            If[load === $Failed, Return[$Failed]];  
            password = GUIKit`GUIRunModal["DatabaseLink/PasswordFrame"];
          ];
      
          If[timeout =!= None, 
            If[!IntegerQ[timeout] || timeout < 0, 
              Message[OpenSQLConnection::timeout, timeout]
            ],
            If[$SQLTimeout =!= None, 
              If[!IntegerQ[timeout] || timeout < 0, 
                Message[OpenSQLConnection::timeout, timeout],
                timeout = $SQLTimeout;
              ],
              timeout = 0;
            ]
          ];
                
          connection = JDBCConnectionManager`getConnection[u, user, password, timeout];
          
          id = ++connectionIndex;
          conn = 
            SQLConnection[
              JDBC[
                driver, 
                url
              ],
              connection,
              id, 
              opts
            ];
          KeepJavaObject[connection];
          AppendTo[$sqlConnections, conn];
          conn
        ]
      ]
    ] 
  ]

OpenSQLConnection[SQLConnection[jdbc_JDBC, opts___Rule], opts2___Rule] := 
  OpenSQLConnection[SQLConnection[jdbc, Null, -1, opts], opts2]

OpenSQLConnection[SQLConnection[
                    jdbc_JDBC,
                    _,
                    _Integer,
                    opts___Rule], 
                  opts2___Rule] := 
  Module[{desc, loc, name, pw, rel, to, un, v},

    {desc, loc, name, pw, rel, to, un, v} = 
      {"Description", "Location", "Name", "Password", 
       "RelativePath","Timeout", "Username", "Version"} 
         /. canonicalOptions[Flatten[{opts2}]] /. canonicalOptions[Flatten[{opts}]] /. Options[OpenSQLConnection];

    OpenSQLConnection[jdbc, "Description"->desc, "Location"->loc, "Name"->name, "Password"->pw,
                            "RelativePath"->rel, "Timeout"->to, "Username"->un, "Version"->v]
  ]

OpenSQLConnection[name_String, opts___Rule] := 
  Module[{list, index = -1},
    list = Flatten[Position[DataSourceNames[], name]];
    If[Length[list] > 0, index = First[list]];
    If[index > 0, 
      OpenSQLConnection[DataSources[][[index]], opts], 
      Message[OpenSQLConnection::notfound, name];
      $Failed
    ]
  ]

CloseSQLConnection[ SQLConnection[
                      _JDBC,
                      connection_,
                      id_Integer,
                      ___Rule]] :=  
                     
  If[JavaObjectQ[connection], 
    If[!connection@isClosed[], 
      connection@close[];
      ReleaseJavaObject[connection];
      $sqlConnections = Drop[ $sqlConnections, 
                              First@Position[ $sqlConnections, 
                                              SQLConnection[_, 
                                                            _, 
                                                            id,
                                                            ___Rule]]];      
    ]
  ]  
                     
(* Table, Column, and DataType Lookup Functionality *)

SQLTableNames[conn_SQLConnection, opts___Rule]:= 
  SQLTableNames[conn, "*", opts];

SQLTableNames[conn_SQLConnection, table_String, opts___Rule] :=
  Module[{tables},
    tables = SQLTables[conn, table, opts];
    If[tables === $Failed, 
      $Failed,
      First /@ tables
    ]
  ]

SQLTables[ conn_SQLConnection, opts___Rule] :=
  SQLTables[ conn, "*", opts]

SQLTables[ conn_SQLConnection, table_String, opts___Rule] :=
  Module[ {data, nameIndex, typeIndex, useOpts, tt, catalog, schema}, 
    useOpts = canonicalOptions[Flatten[{opts}]];
    tt = "TableType" /. useOpts /. Options[ SQLTables ];
    catalog = "Catalog" /. useOpts /. Options[ SQLTables ];
    schema = "Schema" /. useOpts /. Options[ SQLTables ];
    
    data = SQLTableInformation[ conn, table, "Catalog"->catalog, 
                                             "Schema"->schema, 
                                             "TableType"->tt, 
                                             "ShowColumnHeadings"->True];
    If[data === $Failed, Return[$Failed]];
    
    {nameIndex, typeIndex} =
       Flatten[Position[ToUpperCase /@ data[[1]], #] & /@ {"TABLE_NAME", "TABLE_TYPE"}];
       
    data = SQLTable[#[[nameIndex]], "TableType" -> #[[typeIndex]]] & /@ Drop[data, 1]
  ]

SQLTableInformation[ conn_SQLConnection, opts___Rule] :=
  SQLTableInformation[ conn, "*", opts]

SQLTableInformation[ SQLConnection[ _JDBC, connection_, _Integer, ___Rule], 
                     table_String, opts___Rule] :=
  JavaBlock[
    Module[ {data, nameIndex, typeIndex, useOpts, tt, sch, meta, rs, schema, t, 
             catalog}, 
      Block[{$JavaExceptionHandler = ThrowException}, 
        useOpts = canonicalOptions[Flatten[{opts}]];
        catalog = "Catalog" /. useOpts /. Options[ SQLTableInformation ];
        schema = "Schema" /. useOpts /. Options[ SQLTableInformation ];
        tt = "TableType" /. useOpts /. Options[ SQLTableInformation ];
        sch = "ShowColumnHeadings" /. useOpts /. Options[ SQLTableInformation ];

        Which[
          StringQ[tt], tt = {tt}, 
          !ListQ[tt], Message[SQLTables::opt, tt];Return[$Failed]
        ];
        Which[
          catalog === None, catalog = Null, 
          !StringQ[catalog], Message[SQLTables::opt, catalog];Return[$Failed]
        ];
        Which[
          schema === None, schema = Null, 
          !StringQ[schema], Message[SQLTables::opt, schema];Return[$Failed]
        ];
        If[table === "*", t = Null, t = table];      
        If[!JavaObjectQ[connection], 
          Message[SQLConnection::conn];
          Return[$Failed]
        ];
      
        meta = connection@getMetaData[]; 
        rs = meta@getTables[catalog,schema,t,tt];

        LoadJavaClass["com.wolfram.databaselink.SQLStatementProcessor"];
        SQLStatementProcessor`getAllResultData[ rs, False, TrueQ[sch]]
      ]
    ]
  ]
  
SQLColumnNames[conn_SQLConnection, opts___Rule] := 
  SQLColumnNames[conn, {"*", "*"}, opts]
  
SQLColumnNames[conn_SQLConnection, table_String, opts___Rule] := 
  SQLColumnNames[conn, {table, "*"}, opts]

SQLColumnNames[conn_SQLConnection, SQLTable[table_String ,___Rule], opts___Rule] := 
  SQLColumnNames[conn, {table, "*"}, opts]

SQLColumnNames[conn_SQLConnection, SQLColumn[col_String,___Rule ], opts___Rule] := 
  SQLColumnNames[conn, {"*", col}, opts]

SQLColumnNames[conn_SQLConnection, SQLColumn[{table_String, col_String},___Rule ], opts___Rule] := 
  SQLColumnNames[conn, {table, col}, opts]

SQLColumnNames[conn_SQLConnection, {table_String, column_String}, opts___Rule] := 
  Module[{columns},
    columns = SQLColumns[conn, {table, column}, opts];
    If[columns === $Failed, 
      $Failed,
      First /@ columns
    ]
  ]
  
SQLColumns[conn_SQLConnection, opts___Rule] := 
  SQLColumns[conn, {"*", "*"}, opts]
  
SQLColumns[conn_SQLConnection, table_String, opts___Rule] := 
  SQLColumns[conn, {table, "*"}, opts]

SQLColumns[conn_SQLConnection, SQLTable[table_String ,___Rule], opts___Rule] := 
  SQLColumns[conn, {table, "*"}, opts]

SQLColumns[conn_SQLConnection, SQLColumn[col_String,___Rule ], opts___Rule] := 
  SQLColumns[conn, {"*", col}, opts]

SQLColumns[conn_SQLConnection, SQLColumn[{table_String, col_String},___Rule ], opts___Rule] := 
  SQLColumns[conn, {table, col}, opts]

SQLColumns[conn_SQLConnection, {table_String, column_String}, opts___Rule] := 
  Module[ { data, dtn, tableIndex, columnIndex, typeIndex, nullableIndex, lengthIndex, useOpts}, 

    useOpts = canonicalOptions[Flatten[{opts}]];
    catalog = "Catalog" /. useOpts /. Options[ SQLTableInformation ];
    schema = "Schema" /. useOpts /. Options[ SQLTableInformation ];

    data = SQLColumnInformation[ conn, {table, column}, "Catalog"->catalog,
                                                        "Schema"->schema, 
                                                        "ShowColumnHeadings"->True];
    If[data === $Failed, Return[$Failed]];
    
    {tableIndex, columnIndex, typeIndex, nullableIndex, lengthIndex} =
       Flatten[Position[ToUpperCase /@ data[[1]], #] & /@ 
         {"TABLE_NAME", "COLUMN_NAME", "TYPE_NAME", "NULLABLE", "COLUMN_SIZE"}];
       
    data = SQLColumn[{#[[tableIndex]], #[[columnIndex]]}, 
             "DataTypeName" -> #[[typeIndex]],
             "Nullable" -> #[[nullableIndex]],
             "DataLength" -> #[[lengthIndex]]] & /@ Drop[data, 1]       
  ]

SQLColumnInformation[conn_SQLConnection, opts___Rule] := 
  SQLColumnInformation[conn, {"*", "*"}, opts]
  
SQLColumnInformation[conn_SQLConnection, table_String, opts___Rule] := 
  SQLColumnInformation[conn, {table, "*"}, opts]

SQLColumnInformation[conn_SQLConnection, SQLTable[table_String ,___Rule], opts___Rule] := 
  SQLColumnInformation[conn, {table, "*"}, opts]

SQLColumnInformation[conn_SQLConnection, SQLColumn[col_String,___Rule ], opts___Rule] := 
  SQLColumnInformation[conn, {"*", col}, opts]

SQLColumnInformation[conn_SQLConnection, SQLColumn[{table_String, col_String},___Rule ], opts___Rule] := 
  SQLColumnInformation[conn, {table, col}, opts]

SQLColumnInformation[SQLConnection[ _JDBC, connection_, _Integer, ___Rule],
                     {table_String, column_String}, opts___Rule] := 
  JavaBlock[
    Module[ {useOpts, dtn, sch, meta, rs, schema, t, c, catalog}, 
      Block[{$JavaExceptionHandler = ThrowException},

        useOpts = canonicalOptions[Flatten[{opts}]];
        catalog = "Catalog" /. useOpts /. Options[ SQLTableInformation ];
        schema = "Schema" /. useOpts /. Options[ SQLTableInformation ];
        sch = "ShowColumnHeadings" /. useOpts /. Options[ SQLTableInformation ];

        Which[
          catalog === None, catalog = Null, 
          !StringQ[catalog], Message[SQLColumns::opt, catalog];Return[$Failed]
        ];
        Which[
          schema === None, schema = Null, 
          !StringQ[schema], Message[SQLColumns::opt, schema];Return[$Failed]
        ];
        If[table === "*", t = Null, t = table];      
        If[column === "*", c = Null, c = column];      
        If[!JavaObjectQ[connection], 
          Message[SQLConnection::conn];
          Return[$Failed]
        ];
      
        meta = connection@getMetaData[]; 
        rs = meta@getColumns[catalog,schema,t,c];

        LoadJavaClass["com.wolfram.databaselink.SQLStatementProcessor"];
        SQLStatementProcessor`getAllResultData[ rs, False, TrueQ[sch]]
      ]
    ]
  ]

SQLDataTypeNames[ conn_SQLConnection] := 
  Module[ { data, nameIndex }, 
    data = SQLDataTypeInformation[ conn, ShowColumnHeadings->True];
    If[data === $Failed, Return[$Failed]];
    
    {nameIndex} = Flatten[Position[ToUpperCase /@ data[[1]], "TYPE_NAME"]];   
    data = Flatten[#[[nameIndex]] & /@ Drop[data, 1]]
  ]  

SQLDataTypeInformation[SQLConnection[ _JDBC, connection_, _Integer, ___Rule],
                       opts___Rule ] := 
  JavaBlock[
    Module[ {useOpts, sch, meta, rs}, 
      Block[{$JavaExceptionHandler = ThrowException}, 
        useOpts = canonicalOptions[Flatten[{opts}]];
        sch = "ShowColumnHeadings" /. useOpts /. Options[ SQLTableInformation ];

        If[!JavaObjectQ[connection], 
          Message[SQLConnection::conn];
          Return[$Failed]
        ];
      
        meta = connection@getMetaData[]; 
        rs = meta@getTypeInfo[];

        LoadJavaClass["com.wolfram.databaselink.SQLStatementProcessor"];
        SQLStatementProcessor`getAllResultData[ rs, False, TrueQ[sch]]      
      ]
    ]
  ]

SQLTableTypeNames[SQLConnection[ _JDBC, connection_, _Integer, ___Rule]] := 
  JavaBlock[
    Module[ {meta, rs, data}, 
      Block[{$JavaExceptionHandler = ThrowException}, 
        If[!JavaObjectQ[connection], 
          Message[SQLConnection::conn];
          Return[$Failed]
        ];
      
        meta = connection@getMetaData[]; 
        rs = meta@getTableTypes[];

        LoadJavaClass["com.wolfram.databaselink.SQLStatementProcessor"];
        data = SQLStatementProcessor`getAllResultData[ rs, False, False];
        If[data =!= $Failed, data = Flatten[data]];
        data
      ]
    ]
  ]
  
SQLSchemaNames[SQLConnection[ _JDBC, connection_, _Integer, ___Rule]] := 
  JavaBlock[
    Module[ {meta, rs, data}, 
      Block[{$JavaExceptionHandler = ThrowException}, 
        If[!JavaObjectQ[connection], 
          Message[SQLConnection::conn];
          Return[$Failed]
        ];
      
        meta = connection@getMetaData[]; 
        rs = meta@getSchemas[];

        LoadJavaClass["com.wolfram.databaselink.SQLStatementProcessor"];
        data = SQLStatementProcessor`getAllResultData[ rs, False, False];
        If[data =!= $Failed, data = Flatten[data]];
        data      
      ]
    ]
  ]

SQLCatalogNames[SQLConnection[ _JDBC, connection_, _Integer, ___Rule]] := 
  JavaBlock[
    Module[ {meta, rs, data}, 
      Block[{$JavaExceptionHandler = ThrowException},
        If[!JavaObjectQ[connection], 
          Message[SQLConnection::conn];
          Return[$Failed]
        ];
        
        meta = connection@getMetaData[]; 
        rs = meta@getCatalogs[];

        LoadJavaClass["com.wolfram.databaselink.SQLStatementProcessor"];
        data = SQLStatementProcessor`getAllResultData[ rs, False, False];
        If[data =!= $Failed, data = Flatten[data]];
        data
      ]
    ]
  ]

(* SQL Generation Functionality *)

SQLDropTable[ conn_SQLConnection, table:(_SQLTable | _String), opts___Rule] := 
  Module[ { tbl, useOpts, timeout }, 

    useOpts = canonicalOptions[Flatten[{opts}]];
    timeout = "Timeout" /. useOpts /. Options[ SQLExecute ];

    tbl = If[StringQ[table], SQLTable[table], table];
    
    SQLExecute[conn, "DROP TABLE `1`", {tbl}, "Timeout" -> timeout]
  ] 

SQLCreateTable[conn_SQLConnection,
               table:(_SQLTable | _String), 
               col:(_SQLColumn | {__SQLColumn}),
               opts___Rule] := 
  Module[ { tbl, stmt, useOpts, timeout}, 

    useOpts = canonicalOptions[Flatten[{opts}]];
    timeout = "Timeout" /. useOpts /. Options[ SQLExecute ];

    tbl = If[StringQ[table], SQLTable[table], table];

    stmt = "CREATE TABLE `1` ( " <> mingleComma[sqlColumnConvert /@ Flatten[{ col }]] <> ")";
    SQLExecute[conn, stmt, {tbl}, "Timeout" -> timeout]
  ]
          
sqlColumnConvert[ SQLColumn[(name_String | {_String, name_String}), opts___Rule ] ] := 
    Module[ { cn, dt, dtn, dw, nl, useOpts }, 
            useOpts = canonicalOptions[Flatten[{opts}]];
            dtn = "DataTypeName" /. useOpts /. Options[ SQLColumn ]; 
            dw  = "DataLength"   /. useOpts /. Options[ SQLColumn ]; 
            nl  = "Nullable"     /. useOpts /. Options[ SQLColumn ]; 
            
            dt = Which[ Head[ dtn ] === String, " " <> dtn, 
                        True, Message[SQLColumn::datatype];Return[$Failed];
                      ]; 
            dw = Which[ ( dw === None ), "", 
                        ( Head[ dw ] === Integer ), StringJoin@{ "(", ToString[dw], ")" }, 
                        ( True ), Message[ SQLColumn::datalength ];Return[$Failed]; 
                      ]; 
            nl = Which[ ( nl === None ), "", 
                        ( nl === True ), " NULL",
                        ( nl === False ), " NOT NULL", 
                        ( True ), ""
                      ]; 
            stmt = StringJoin@{ name, dt, dw, nl }         
          ]

SQLInsert[conn_SQLConnection,
          table:(_SQLTable | _String),
          names:{___String}, 
          values:{__},
          opts___Rule] :=
  SQLInsert[conn, table, SQLColumn/@names, values]

SQLInsert[conn_SQLConnection,
          table:(_SQLTable | _String),
          names:{{__String,_String}...}, 
          values:{__},
          opts___Rule] :=
  SQLInsert[conn, table, SQLColumn[Last[#]]&/@names, values]

SQLInsert[conn_SQLConnection,
          (SQLTable[table_String, ___Rule] | table_String),
          names:{ SQLColumn[(_String|{_String, _String}), ___Rule] ... }, 
          values:{__},
          opts___Rule] := 
  Module[ { useOpts, timeout, tbl, cols, vals },

    useOpts = canonicalOptions[Flatten[{opts}]];
    timeout = "Timeout" /. useOpts /. Options[ SQLExecute ];

    tbl = If[StringQ[table], tbl, First[table]];    

    cols = 
      If[names === {},
        "",
        StringJoin[" (", mingleComma[getSQLColumnName /@ names], ")"]
      ];
    
    vals = 
      If[MatchQ[values, {__List}],
        StringJoin["(", mingleComma["?" & /@ Range[Length[First[values]]]] ,")"], 
        StringJoin["(", mingleComma["?" & /@ Range[Length[values]]] ,")"]
      ];

    SQLExecute[conn, "INSERT INTO " <> table <> cols <> " VALUES " <> vals, values, "Timeout"->timeout]
  ]

getSQLColumnName[SQLColumn[(col_String|{table_String,col_String}),opts___Rule]]:=
  Module[{nm},
    nm= If[Head[Unevaluated[table]] =!= String, col, table <> "." <> col];
    stripSpaces@If[areSpaces[nm], "\"" <> nm <> "\"", nm]
  ]

          
SQLSelect[conn_SQLConnection | conn_String,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          opts___Rule
         ] := 
  SQLSelect[conn, table, SQLColumn["*"], None, opts];
  
SQLSelect[conn_SQLConnection | conn_String,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          columns:(_SQLColumn | {__SQLColumn}),
          opts___Rule] :=
  SQLSelect[conn, table, Flatten@{columns}, None, opts];

SQLSelect[conn_SQLConnection | conn_String,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          columns:(_String | {__String}),
          opts___Rule] :=
  SQLSelect[conn, table, SQLColumn/@(Flatten@{columns}), None, opts];

SQLSelect[conn_SQLConnection | conn_String,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          columns:{{_String, _String}..},
          opts___Rule] :=
  SQLSelect[conn, table, SQLColumn/@columns, None, opts];

SQLSelect[conn_SQLConnection | conn_String,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          condition_,
          opts___Rule] := 
  SQLSelect[conn, table, SQLColumn["*"], condition, opts];

SQLSelect[conn_SQLConnection,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          columns:(_SQLColumn | {__SQLColumn}),
          condition_,
          opts___Rule] :=
  Module[ { tbls, cols, useOpts, distinct, orderby, order,
            maxrows, timeout, gas, sch, where, stmt}, 
  
    tbls = 
      If[!MatchQ[Flatten@{table}, {__SQLTable}], 
        SQLArgument @@ (SQLTable/@Flatten@{table}),
        SQLArgument @@ Flatten@{table}
      ];
    cols = SQLArgument @@ Flatten@{columns};
            
    useOpts  = canonicalOptions[Flatten[{opts}]];
    distinct = "Distinct"         /. useOpts /. Options[ SQLSelect ]; 
    order    = "SortingColumns"   /. useOpts /. Options[ SQLSelect ];
    maxrows  = "MaxRows" /. useOpts /. Options[ SQLExecute ];
    timeout  = "Timeout" /. useOpts /. Options[ SQLExecute ];
    gas      = "GetAsStrings" /. useOpts /. Options[ SQLExecute ];
    sch      = "ShowColumnHeadings" /. useOpts /. Options[ SQLExecute ];
    rrs      = "ResultSet" /. useOpts /. {"ResultSet"->False};
    
            
    distinct = 
      If[ ( distinct === True ), 
        "DISTINCT ", 
        ""
      ]; 
    orderby = 
      If[ ( order  === None ), 
        order = {};
        "",          
        " ORDER BY `4`"
      ]; 
    where = 
      If[condition === None,
        "",
        " WHERE `3`"
      ]; 

    stmt = "SELECT " <> distinct <> "`1` FROM `2`" <> where <> orderby;
    
    SQLExecute[conn, stmt, {cols, tbls, condition, SQLArgument @@ Flatten[{order}]}, 
      "MaxRows"->maxrows, "Timeout"->timeout, "GetAsStrings"->gas, 
      "ShowColumnHeadings"->sch, "ResultSet"->rrs]
  ]

SQLSelect[conn_SQLConnection | conn_String,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          columns:(_String | {__String}),
          condition_,
          opts___Rule] :=
  SQLSelect[conn, table, SQLColumn/@(Flatten@{columns}), condition, opts];

SQLSelect[conn_SQLConnection | conn_String,
          table:(_SQLTable | {__SQLTable} | _String | {__String}),
          columns:{{_String, _String}..},
          condition_,
          opts___Rule] :=
  SQLSelect[conn, table, SQLColumn/@columns, condition, opts];

SQLDelete[ conn_SQLConnection,
           table:(_SQLTable | {__SQLTable} | _String | {__String}), 
           opts___Rule] := 
  SQLDelete[ conn, table, None, opts] 

SQLDelete[ conn_SQLConnection,
           table:(_SQLTable | _String), 
           condition_, 
           opts___Rule] := 
  Module[ { useOpts, timeout, tbl, where }, 

    useOpts = canonicalOptions[Flatten[{opts}]];
    timeout = "Timeout" /. useOpts /. Options[ SQLExecute ];

    tbl = If[StringQ[table], SQLTable[table], table];

    where = 
      If[condition === None,
        "",
        " WHERE `2`"
      ]; 
      
    SQLExecute[conn, "DELETE FROM `1`" <> where, {tbl, condition}, "Timeout"->timeout]
  ]

SQLUpdate[conn_SQLConnection,
          table:(_SQLTable | _String),
          names:{__String}, 
          values:{__},
          opts___Rule] :=
  SQLUpdate[conn, table, SQLColumn/@names, values, None, opts]

SQLUpdate[conn_SQLConnection,
          table:(_SQLTable | _String),
          names:{{__String,_String}..}, 
          values:{__},
          opts___Rule] :=
  SQLUpdate[conn, table, SQLColumn[Last[#]]&/@names, values, None, opts]

SQLUpdate[conn_SQLConnection,
          table:(_SQLTable | _String),
          names:{__SQLColumn }, 
          values:{ __ },
          opts___Rule] := 
  SQLUpdate[conn, table, names, values, None, opts]


SQLUpdate[conn_SQLConnection,
          table:(_SQLTable | _String),
          names:{__String}, 
          values:{__},
          condition_,
          opts___Rule] :=
  SQLUpdate[conn, table, SQLColumn/@names, values, condition, opts]

SQLUpdate[conn_SQLConnection,
          table:(_SQLTable | _String),
          names:{{__String,_String}..}, 
          values:{__},
          condition_,
          opts___Rule] :=
  SQLUpdate[conn, table, SQLColumn[Last[#]]&/@names, values, condition, opts]

SQLUpdate[ conn_SQLConnection,
           table:(_SQLTable | _String), 
           names:{__SQLColumn}, 
           values:{__},
           condition_,
           opts___Rule
         ] := 
  Module[ { useOpts, timeout, tbl, where, set }, 
  
    useOpts = canonicalOptions[Flatten[{opts}]];
    timeout = "Timeout" /. useOpts /. Options[ SQLExecute ];

    tbl = If[StringQ[table], SQLTable[table], table];
    
    where = 
      If[condition === None,
        "",
        " WHERE `3`"
      ]; 
      
    set = MapThread[Rule[#1, #2] &, {names, values}];
    
    SQLExecute[conn, "UPDATE `1` SET `2`" <> where, {tbl, SQLArgument @@ set, condition}, "Timeout"->timeout] 
  ]

formatSQL[stmt_String, params_List]:=
  formatSQL[stmt, params, Range[Length[params]]]

formatSQL[stmt_String, params_List, indexes_List]:=
  Module[{newStmt = stmt, newParams = {}, j},  
    For[j = 1, j <= Length[indexes], j++,
      {newStmt, newParams} = setSQLValue[indexes[[j]], params[[indexes[[j]]]], newStmt, newParams];
    ];
    {newStmt, newParams}
  ]

setSQLValue[i_Integer, 
            (val_Integer | val_Real | val_String | val:True | val:False | 
             val:Null | val_SQLBinary | val_SQLDateTime | val_SQLExpr), 
            stmt_String, 
            params_List]:=
  {StringReplace[stmt, {"`" <> ToString[i] <> "`" -> "?"}], Append[params, val]}

setSQLValue[i_Integer, SQLTable[name_String,opts___Rule], stmt_String, params_List]:=
    {StringReplace[stmt, {"`" <> ToString[i] <> "`" -> name}], params}

setSQLValue[i_Integer,
            SQLColumn[(col_String|{table_String,col_String}),opts___Rule],
            stmt_String,
            params_List]:=
  {StringReplace[stmt, {"`" <> ToString[i] <> "`" -> 
     If[Head[Unevaluated[table]] =!= String, 
       col, 
       table <> "." <> col
     ]}],
   params}

setSQLValue[i_Integer, list_SQLArgument, stmt_String, params_List]:=
  setSQLComplexValue[i, 
                       StringJoin[mingleComma["`" <> ToString[#] <> "`" & /@ Range[1, Length[list]]]], 
                       List@@list, 
                       stmt, 
                       params]

setSQLValue[i_Integer, 
            ( Power[(col:SQLColumn[(_String | {_String, _String}), opts___Rule]), -1 ] ), 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(1/`1`)", {col}, stmt, params]

setSQLValue[i_Integer, 
            ( (col:SQLColumn[(_String | {_String, _String}), opts___Rule]) * -1 ), 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(-`1`)", {col}, stmt, params]

setSQLValue[i_Integer, 
            ( Rational[ 1, den_ ] * (col:SQLColumn[(_String | {_String, _String}), opts___Rule]) ), 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(`1`/`2`)", {col, den}, stmt, params]

setSQLValue[i_Integer, left_ * right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` * `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ + right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` + `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ - right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` - `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ && right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` AND `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ || right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` OR `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, 
            (col:SQLColumn[(_String | {_String, _String}), opts___Rule]) == Null, 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(`1` IS NULL)", {col}, stmt, params]

setSQLValue[i_Integer, 
            ( v1_ <= (col:SQLColumn[(_String | {_String, _String}), opts___Rule]) <= v2_ ), 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(`1` BETWEEN `2` AND `3`)", {col, v1, v2}, stmt, params]

setSQLValue[i_Integer, 
            ( v1_ >= (col:SQLColumn[(_String | {_String, _String}), opts___Rule]) >= v2_ ), 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(`1` BETWEEN `2` AND `3`)", {col, v2, v1}, stmt, params]

setSQLValue[i_Integer, 
            ( v1_ < (col:SQLColumn[(_String | {_String, _String}), opts___Rule]) < v2_ ), 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(`1` > `2` AND `1` < `3`)", {col, v1, v2}, stmt, params]

setSQLValue[i_Integer, 
            ( v1_ > (col:SQLColumn[(_String | {_String, _String}), opts___Rule]) > v2_ ), 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "(`1` > `2` AND `1` < `3`)", {col, v2, v1}, stmt, params]

setSQLValue[i_Integer, !val_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(NOT `1`)", {val}, stmt, params]

setSQLValue[i_Integer, left_ == right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` = `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ != right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` != `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ > right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` > `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ < right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` < `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ >= right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` >= `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, left_ <= right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` <= `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, SQLStringMatchQ[left_SQLColumn, right_String], stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` LIKE `2`)", {left, right}, stmt, params]

setSQLValue[i_Integer, SQLMemberQ[left_List, right_SQLColumn], stmt_String, params_List]:=
  setSQLComplexValue[i, "(`1` IN (`2`))", {right, SQLArgument@@left}, stmt, params]

setSQLValue[i_Integer, 
            col:SQLColumn[(_String | {_String, _String}), opts___Rule] -> "Ascending", 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "`1` ASC", {col}, stmt, params]

setSQLValue[i_Integer, 
            col:SQLColumn[(_String | {_String, _String}), opts___Rule] -> "Descending", 
            stmt_String, 
            params_List]:=
  setSQLComplexValue[i, "`1` DESC", {col}, stmt, params]

setSQLValue[i_Integer, left_ -> right_, stmt_String, params_List]:=
  setSQLComplexValue[i, "`1` = `2`", {left, right}, stmt, params]

setSQLValue[i_Integer, x_, stmt_String, params_List]:=
  (Message[SQLValue::illegalvalue, x ];
   Throw[$Failed])
  
setSQLValue[x___]:= 
  (Message[SQLValue::illegalvalue, { x }];
   Throw[$Failed])

setSQLComplexValue[i_Integer, localStmt_String, localParams_List, stmt_String, params_List] :=
  Module[{str, listParams = {}},
    {str, listParams} = formatSQL[ localStmt, localParams];
    {StringReplace[stmt, {"`" <> ToString[i] <> "`" -> str}], Join[params, listParams]}
  ]

mingleObject[lst_List, object_] := 
  If[lst === {}, 
    {}, 
    Drop[Flatten[Thread[{lst, object}, List, 1], 1, List], -1]
  ]
    
mingleComma[ lst_List ] := 
    mingleObject[ lst, ", " ];

mingleComma[ lst_ ] := 
    mingleObject[ {lst}, ", " ];              

areSpaces[str_String] := Or@@((# === " ") &/@(Characters@str))

stripSpaces[ str_String ] := 
    FixedPoint[ If[ StringTake[#, 1] === " ", 
                    StringDrop[#, 1], #
                  ]&, 
                FixedPoint[ If[ StringTake[#, -1] === " ", 
                                StringDrop[#, -1], #
                              ]&, 
                            str
                          ]
              ]

(*===================================================================*)
(*================== FORMATTING AND CONVERSION ======================*)
(*===================================================================*)

Format[ SQLResultSet[ i_Integer, rs_?JavaObjectQ ] ] := 
  Module[{forward = False}, 
    If[rs@getType[] == 1003, 
      SQLResultSet[ i, "<>", "ForwardOnly" ], 
      SQLResultSet[ i, "<>", "Scrollable" ]
    ]
  ]

Format[ SQLConnection[ _JDBC, conn_, id_Integer, opts___Rule]] := 
  Module[{name}, 
    name = "Name" /. canonicalOptions[Flatten[{opts}]] /. Options[ SQLConnection ]; 
    If[JavaObjectQ[conn], 
      If[conn =!= Null && !conn@isClosed[], 
        SQLConnection[ name, id, "Open", "<>" ],
        SQLConnection[ name, id, "Closed", "<>" ]
      ],
      SQLConnection[ name, id, "Closed", "<>" ]
    ]
  ]  

Format[ SQLSavepoint[_, opts___Rule]] := 
  Module[{name}, 
    name = "Name" /. canonicalOptions[Flatten[{opts}]] /. Options[ SQLSavepoint ]; 
    SQLSavepoint[ name, "<>" ]
  ]  

(*===================================================================*)
(*================= SQLExecute ======================================*)
(*===================================================================*)

SQLExecute[conn_SQLConnection, sql_String, opts___Rule] :=
  Module[{query, pos},
    If[MemberQ[SQLQueryNames[], sql], 
      pos = First[Flatten[Position[SQLQueryNames[], sql]]];
      query = SQLQueries[][[pos]];
      SQLExecute[conn, query],
      SQLExecute[ conn, sql, {}, opts]
    ]
  ];
  
SQLExecute[ SQLConnection[ _JDBC, connection_, _Integer, ___Rule], ps_String, args:{___}, opts___Rule] :=
  JavaBlock[
    Module[{sql = ps, params, stmt,i, rs, useOpts, maxrows, 
            timeout, gas, sch, rrs, rsc = 1007, rst = 1003, 
            posList, sortedPosList, indexes, idx = 1, 
            stringPosList, somePosList, otherPosList},
      Block[{$JavaExceptionHandler = ThrowException},
        Catch[
                
          If[!JavaObjectQ[connection], 
            Message[SQLConnection::conn];
            Return[$Failed]
          ];
          
          useOpts = canonicalOptions[Flatten[{opts}]];
          maxrows = "MaxRows" /. useOpts /. Options[ SQLExecute ];
          timeout = "Timeout" /. useOpts /. Options[ SQLExecute ];
          gas     = "GetAsStrings" /. useOpts /. Options[ SQLExecute ];
          sch     = "ShowColumnHeadings" /. useOpts /. Options[ SQLExecute ];
          rrs     = "ResultSet" /. useOpts /. {"ResultSet"->False};

          If[maxrows =!= Automatic && maxrows =!= All, 
            If[maxrows === None, maxrows = 0];
            If[!IntegerQ[maxrows] || maxrows < 0, 
              Message[SQLExecute::maxrows, maxrows]
            ],
            maxrows = 0;
          ];

          If[timeout =!= None, 
            If[!IntegerQ[timeout] || timeout < 0, 
              Message[SQLExecute::timeout, timeout]
            ],
            If[$SQLTimeout =!= None, 
              If[!IntegerQ[timeout] || timeout < 0, 
                Message[SQLExecute::timeout, timeout],
                timeout = $SQLTimeout;
              ],
              timeout = 0;
            ]
          ];
          
          If[MatchQ[rrs, {_String}], 
            rst = 
              Switch[
                First[rrs],
                "ForwardOnly", 1003, 
                "ScrollInsensitive", 1004,
                "ScrollSensitive", 1005,
                _, Message[SQLResultSetOpen::mode, First[rrs]];Return[$Failed]];                              
            rrs = True,
            rrs = False
          ];
          
          (* Check to see if there is more than one item to execute *)
          If[MatchQ[args, {__List}],

            (* Check to see if anything needs to be replaced *) 
            posList = StringPosition[ps, "`" <> ToString[#] <> "`"] & /@Range[Length[First[args]]];
            otherPosList = StringPosition[ps, "``"];
            AppendTo[posList, otherPosList];
            If[Flatten[posList] === {}, 
              sql = ps;
              params = args,
        
              (* If things need to be replaced, get a list of indexes for each replacement. 
                 Then replace the values and get a set of parameters that can be handled by Java *)
              sortedPosList = Sort[Flatten[posList, 1]];
              indexes = (First[Flatten[Position[posList,#]]] & /@sortedPosList); 
              somePosList = Flatten[Position[indexes, Length[args] + 1]];
              indexes = If[# === Length[args] + 1, idx++; idx - 1, idx = # + 1; #] & /@ indexes;
              stringPosList = "`" <> ToString[indexes[[#]]] <> "`" & /@ somePosList;              
              {sql, params} = Transpose[formatSQL[StringReplacePart[ps, stringPosList, otherPosList], #, indexes] & /@ args];
              sql = First[sql];       
            ];

            ,
            
            (* There is only one item to execute check to see if anything needs replaced *)
            posList = StringPosition[ps, "`" <> ToString[#] <> "`"] & /@Range[Length[args]];            
            otherPosList = StringPosition[ps, "``"];
            AppendTo[posList, otherPosList];
            If[Flatten[posList] === {}, 
      
              (* If nothing needs to be replaced just return the statement and params as is *)
              sql = ps;
              params = {args},
          
              (* If things need to be replaced, get a list of indexes for each replacement. 
                 Then replace the values and get a set of parameters that can be handled by Java *)
              sortedPosList = Sort[Flatten[posList, 1]];
              indexes = (First[Flatten[Position[posList,#]]] & /@sortedPosList);    
              somePosList = Flatten[Position[indexes, Length[args] + 1]];
              indexes = (If[# === Length[args] + 1, idx++; idx - 1, idx = # + 1; #] & /@ indexes);
              stringPosList = "`" <> ToString[indexes[[#]]] <> "`" & /@ somePosList;
              {sql, params} = formatSQL[StringReplacePart[ps, stringPosList, otherPosList], args, indexes];
              params = {params};
            ];
          ];
          
          (* Convert SQLExpr objects to strings ahead of time. *) 
          params = params /. {SQLExpr[x_] :> SQLExpr[ToString[FullForm[x]]]};
                       
          (* Execute the JDBC prepared statement with parameters that can be handled by Java *)
          LoadJavaClass["com.wolfram.databaselink.SQLStatementProcessor"];
          result = SQLStatementProcessor`processSQLStatement[
            connection, sql, params, maxrows, timeout, TrueQ[gas], TrueQ[sch], TrueQ[rrs], rst, rsc];
            
          If[MatchQ[result, {_Integer}], 
            First[result],
            If[MatchQ[result, {_?JavaObjectQ}], 
              KeepJavaObject[First[result]];
              First[result],
              result
            ]
          ]
        ]
      ]
    ]
  ]
  
SQLExecute[queryName_String] :=
  Module[{query, pos},
    If[MemberQ[SQLQueryNames[], queryName], 
      pos = First[Flatten[Position[SQLQueryNames[], queryName]]];
      query = SQLQueries[][[pos]];
    ];    
    SQLExecute[query]
  ]

SQLExecute[SQLSelect[connName_String,
                     table:(_SQLTable | {__SQLTable} | _String | {__String}),
                     columns:(_SQLColumn | {__SQLColumn}),
                     condition_,
                     opts___Rule]] :=
  Module[{conn = OpenSQLConnection[connName], data = {}},
    data = SQLSelect[conn, table, columns, condition, opts];
    CloseSQLConnection[conn];
    data
  ]

SQLExecute[conn_SQLConnection, 
           SQLSelect[connName_String,
                     table:(_SQLTable | {__SQLTable} | _String | {__String}),
                     columns:(_SQLColumn | {__SQLColumn}),
                     condition_,
                     opts___Rule]] :=
  SQLSelect[conn, table, columns, condition, opts]
  
(*===================================================================*) 
(*========================== ResultSets  ============================*) 
(*===================================================================*) 

resultSetIndex = 0;

SetAttributes[ SQLResultSetOpen, HoldFirst]

SQLResultSetOpen[ (s_SQLSelect | s_SQLExecute), opts___Rule] :=
  Module[{resultSet, useOpts, mode, conc}, 
 
    useOpts = canonicalOptions[Flatten[{opts}]];
    mode = "Mode" /. useOpts /. Options[ SQLResultSetOpen ]; 
  
    resultSet = ReleaseHold[Insert[Hold[s], "ResultSet" -> {mode}, {1, -1}]];
    If[!JavaObjectQ[resultSet] || resultSet === $Failed, Return[$Failed]];
    resultSet = SQLResultSet[resultSetIndex++, resultSet];
    AppendTo[$resultSets, resultSet];
    resultSet
  ]

SQLResultSetClose[ SQLResultSet[ id_Integer, rs_?JavaObjectQ ] ] := 
  Module[ {}, 
    If[ ( MemberQ[ $resultSets, SQLResultSet[ id, rs ] ] ), 
      $resultSets = Drop[$resultSets, First@Position[$resultSets, SQLResultSet[id, rs]]]; 
      rs@close[];
      ReleaseJavaObject[rs];
    ] 
  ] 
  
SQLResultSetCurrent[ SQLResultSet[ _Integer, rs_?JavaObjectQ], opts___Rule ] := 
  Module[ {useOpts, gas = False, results},
    Block[{$JavaExceptionHandler = ThrowException},     
      Catch[
        useOpts = canonicalOptions[Flatten[{opts}]];
        gas = "GetAsStrings" /. useOpts /. Options[ SQLResultSetCurrent ]; 
  
        results = SQLStatementProcessor`getLimitedResultData[0, rs, gas];
        If[ListQ[results] && Length[results] > 0, First[results], results]
      ]
    ]
  ]

SQLResultSetRead[ rs_SQLResultSet, opts___Rule ] := 
  Module[{results}, 
    results = SQLResultSetRead[rs, 1, opts];
    If[ListQ[results] && Length[results] > 0, First[results], results]
  ]

SQLResultSetRead[ rs_SQLResultSet, 0, opts___Rule ] := {}

SQLResultSetRead[ SQLResultSet[ _Integer, rs_?JavaObjectQ], limit_Integer, opts___Rule] := 
  Module[ {useOpts, gas = False},
    Block[{$JavaExceptionHandler = ThrowException},     
      Catch[
        useOpts = canonicalOptions[Flatten[{opts}]];
        gas = "GetAsStrings" /. useOpts /. Options[ SQLResultSetRead ]; 
  
        If[limit < 0 && rs@getType[] == 1003,
          Message[SQLResultSet::forwardonly];
          Return[$Failed];
        ];            
        SQLStatementProcessor`getLimitedResultData[limit, rs, gas]
      ]
    ]
  ]

SQLResultSetTake[ rs_SQLResultSet, {start_}, opts___Rule] :=
  SQLResultSetTake[rs, {start, start}, opts]  

SQLResultSetTake[ SQLResultSet[ _Integer, rs_?JavaObjectQ], {start_Integer, end_Integer}, opts___Rule] :=
  Module[ {useOpts, gas = False, eRow = 0, sRow = 0, diff, current},
    Block[{$JavaExceptionHandler = ThrowException},     
      Catch[
        current = rs@getRow[];
        
        If[rs@getType[] == 1003,
          Message[SQLResultSet::forwardonly];
          Return[$Failed];
        ];            

        useOpts = canonicalOptions[Flatten[{opts}]];
        gas = "GetAsStrings" /. useOpts /. Options[ SQLResultSetTake ]; 
        
        If[start == 0, 
          Message[SQLResultSetTake::invalidrange, {start, end}];Return[$Failed]
        ];

        If[end =!= 0, 
          If[!rs@absolute[end], 
            resetCursor[rs, current];
            Message[SQLResultSetTake::invalidrange, {start, end}];
            Return[$Failed]
          ];
          eRow = rs@getRow[], 
          Message[SQLResultSetTake::invalidrange, {start, end}];
          Return[$Failed]
        ];

        If[!rs@absolute[start],
          resetCursor[rs, current];
          Message[SQLResultSetTake::invalidrange, {start, end}];
          Return[$Failed];
        ];
        sRow = rs@getRow[];
        
        diff = eRow - sRow;
        Which[
          diff < 0, 
            diff = diff - 1;
            If[!rs@relative[1], rs@afterLast[]],
          diff > 0, 
            diff = diff + 1;
            If[!rs@relative[-1], rs@beforeFirst[]]
        ];

        SQLStatementProcessor`getLimitedResultData[diff, rs, gas]
      ]
    ]
  ]

resetCursor[rs_, position_Integer] :=
  Module[{}, 
    If[!rs@absolute[position], 
      If[position == 0, rs@beforeFirst[], rs@afterLast[]]
    ]
  ]

SQLResultSetShift[ SQLResultSet[ _Integer, rs_?JavaObjectQ], rows_Integer] := 
  Module[ {valid = False},
    Block[{$JavaExceptionHandler = ThrowException},     
      Catch[
        If[rs@getType[] == 1003,
          If[rows < 0, 
            Message[SQLResultSet::forwardonly];
            Return[$Failed];
          ];
          For[i = 0, i < rows, i++, valid = rs@next[]]; 
          valid,
          rs@relative[rows]        
        ]
      ]
    ]
  ]
  
SQLResultSetGoto[ SQLResultSet[ _Integer, rs_?JavaObjectQ], row_Integer | row:Infinity] := 
  Block[{$JavaExceptionHandler = ThrowException},     
    Catch[
      If[rs@getType[] == 1003,
        Message[SQLResultSet::forwardonly];
        Return[$Failed];
      ];   
      Switch[row, 
        0, rs@beforeFirst[];False,
        Infinity, rs@afterLast[];False, 
        _, rs@absolute[row]
      ] 
    ]
  ]

SQLResultSetPosition[ SQLResultSet[ _Integer, rs_?JavaObjectQ]] := 
  Block[{$JavaExceptionHandler = ThrowException},     
    Catch[rs@getRow[]]]
    
SQLResultSetColumnNames[ SQLResultSet[ _Integer, rs_?JavaObjectQ]] :=
  Block[{$JavaExceptionHandler = ThrowException},     
    Catch[
      SQLStatementProcessor`getHeadings[ rs, True]
    ]
  ]
  

(*===================================================================*) 
(*======================== Transactions =============================*) 
(*===================================================================*) 

$inTransaction = False;

SQLBeginTransaction[SQLConnection[ _JDBC, connection_, _Integer, ___Rule]] :=
  (If[!JavaObjectQ[connection], 
     Message[SQLConnection::conn];
     Return[$Failed]
   ];    
   If[$inTransaction,
     Message[SQLBeginTransaction::nested],
     $inTransaction = True;
     connection@setAutoCommit[False]
   ])

SQLCommitTransaction[SQLConnection[ _JDBC, connection_, _Integer, ___Rule]] :=
  (If[!JavaObjectQ[connection], 
     Message[SQLConnection::conn];
     Return[$Failed]
   ];    
   connection@commit[];
   $inTransaction = False;
   connection@setAutoCommit[True];)

SQLRollbackTransaction[SQLConnection[ _JDBC, connection_, _Integer, ___Rule]] :=
  (If[!JavaObjectQ[connection], 
     Message[SQLConnection::conn];
     Return[$Failed]
   ];    
   connection@rollback[];
   $inTransaction = False;
   connection@setAutoCommit[True];)

SQLRollbackTransaction[SQLConnection[ _JDBC, connection_, _Integer, ___Rule], 
                       SQLSavepoint[savepoint_, opts___Rule]] :=
  (If[!JavaObjectQ[connection], 
     Message[SQLConnection::conn];
     Return[$Failed]
   ];    
   If[!JavaObjectQ[savepoint], 
     Message[SQLRollbackTransaction::savepoint];
     Return[$Failed]
   ];    
   connection@rollback[savepoint];)

SQLSetSavepoint[SQLConnection[ _JDBC, connection_, _Integer, ___Rule]] :=
  (If[!JavaObjectQ[connection], 
     Message[SQLConnection::conn];
     Return[$Failed]
   ];    
   Check[SQLSavepoint[connection@setSavepoint[]], 
     (Message[SQLSetSavepoint::version];$Failed), JLink`Java::nometh])

SQLSetSavepoint[SQLConnection[ _JDBC, connection_, _Integer, ___Rule], name_String] :=
  (If[!JavaObjectQ[connection], 
     Message[SQLConnection::conn];
     Return[$Failed]
   ];    
   Check[SQLSavepoint[connection@setSavepoint[name], "Name"->name], 
     (Message[SQLSetSavepoint::version];$Failed), JLink`Java::nometh])

(*===================================================================*) 
(*====================== Stored Procedures ==========================*) 
(*===================================================================*) 

(*
InstallStoredProcedures[SQLConnection[ _JDBC, connection_, _Integer, ___Rule],
                        context_String] :=
  Module[{meta, procedures},

    If[!StringMatchQ[context, "*`"], 
      Message[
        InstallService::"context", 
        context, 
        "contexts must end with a '`'"];
      Return[$Failed];
    ];
  
    If[!MatchQ[
         StringPosition[
           context, 
           {".", "_", "~", "!", "@", "#", "$", "%", "^", 
            "&", "*", "(", ")", "-", "+", "=", "{", "[", 
            "}", "]", "|", "\\", ":", ";", "\"", "\'", 
            "<", ",", ">", "?", "/", " "}], 
         {}], 
      Message[
        InstallService::"context", 
        context, 
        "contexts must be alpha-numeric."];
      Return[$Failed];
    ];
 
    positions = 
      Drop[Prepend[(First[#] + 1) & 
        /@ StringPosition[context, "`"], 1], -1];
    test = (If[DigitQ[StringTake[context, {#,#}]], $Failed] & /@ positions);
    If[Length[Cases[test, $Failed]] > 0, 
      Message[
        InstallService::"context", 
        context, 
        "contexts must not begin with a digit."];
      Return[$Failed];
    ];
    
    meta = connection@getMetaData[];
    procedures = meta @getProcedures[Null, Null, Null];
    SQLStatementProcessor`getResultData[procedures, False, False]
  ]
*)
(*===================================================================*) 
(*============================= GUI =================================*) 
(*===================================================================*) 

DatabaseExplorer[] := 
  Module[{load},
    load = 
      Check[
        Needs["GUIKit`"], 
        Message[DatabaseExplorer::nogui]; $Failed, 
        Get::noopen, Needs::nocont];
    If[load === $Failed, Return[$Failed]];
    GUIKit`GUIRun["DatabaseExplorer"]
  ]

End[  ] (* Database`SQL`Private` *) 
