(* :Name: Calculus`DSolveIntegrals` *)

(* :Title: Differential Invariants for ODEs, 
    Complete Integrals for Nonlinear
    First-order Partial Differential Equations *)

(* :Author: Alexei V. Bocharov *)

(* :Context: Calculus`DSolveIntegrals` *)

(* :Package Version: 3.0 *)

(* :Copyright: Copyright 1992-2005,  Wolfram Research, Inc.
*)

(* :History:
       Originally delivered by A.V. Bocharov in August 1992.
       First   bug   fixes  by  A.V.Bocharov  in  May  1993.
       Most current bug fixes by A.V.Bocharov in March 1995.
       Revamped by A.V.Bocharov in preparation for autoloading
       as a last-minute 3.0 assignment from S.W. October 1995.
*)

(* :Keywords:  *)

(* :Source:
	A. Bocharov, Symbolic Mathematica Solvers for Nonlinear
                    Differential Equations, Mathematica Journal, vol 3,
	            issue 2,1993,PP.63-69.
*)

(* :Summary:
This package extends the functionality of DSolve,
enabling the function to handle first-order partial
differential equations.
*)

(* :Mathematica Version: 3.0 *)

(*****************************************************************************)

(* Imported symbol *)

   DSolve`PDEAppl1

BeginPackage["Calculus`DSolveIntegrals`"]

CompleteIntegral::usage =
"CompleteIntegral[eqn, u[x, y, ...], {x, y, ...}] builds a complete
integral for the nonlinear first-order differential equation eqn in the
unknown function u with independent variables x, y, ... ."

(*
DifferentialInvariants::usage =
"DifferentialInvariants[{eqn1, eqn2, ...}, {u[x], v[x], ...}, x] gives a list of constants of motion (differential invariants)first of the system of ordinary differential equations in unknown variables u, v, ... ."
*)

IntegralConstants::usage =
"IntegralConstants is obsolete, please use GeneratedParameters instead."
IntegralConstants = GeneratedParameters;

B::usage =  
"B[i] is the default form for the i-th parameter produced in building a
complete integral for a differential equation with CompleteIntegral"

Begin["`Private`"]
 
DifferentialInvariants[args__]:=
   Module[ {ans},
           ans=pFirstIntegrals[args];
	   ans[[1]]/;ans=!=Infinity
         ] (*eludom*)
	 
CompleteIntegral[equ_,dvar_,ivars_]:=
   CompleteIntegral[equ,dvar,ivars,GeneratedParameters->B]
	 
CompleteIntegral[equ_,dvar_,ivars_,GeneratedParameters->b_]:=
   Module[ {ans},
           ans = pCompleteIntegral[equ,dvar,ivars,GeneratedParameters->b];
	   ans/; FreeQ[ans,pCompleteIntegral]
	 ] (*eludom*)
	 
HoldPattern[PDEAppl1[ lhs_==rhs_ , zz_[vars__] , {vars__} ]] :=
   Module[ {w1,w0,i=1,ok,eq},
           eq=lhs-rhs;
           If[ Length[{vars}]<2 ,
	       False,
	       (*else*)
	       w0=Union[Cases[{eq},zz[a__],-1]];
	       If[ (w0==={}) || (w0=={zz[vars]}),
	           (*then*)
	            DerivativeList=Union[Cases[{eq},Derivative[mu__][zz][a__],-1]];
		    ok=Length[DerivativeList]>0;
		    While[ (i<=Length[DerivativeList]) && ok ,
		           If[ ((Head[Head[DerivativeList[[i]]]]/.
			        Derivative:>Plus)===1),
			       ok = (DerivativeList[[i]]/.Derivative[mu__][zz]:>List)
			            === {vars};
			       If[ ok, i++, Null]
			       	    ,
			       (*else*)
			       ok=False
			     ] (*fi*)  
		         ] (*elihw*)
			 ;
	            ok		 
		    ,
		   (*else*)
             Null    ] (* fi *)		    
	     ] (*fi*)      
         ] (*eludom*)		     

HoldPattern[PDEAppl1[other___]]:=False

pFirstIntegrals[ dsystem_List , {vars__}, zz_ ] := 
			    (* Build a set of first integrals     *)
			    (* for the system of first-order      *)
			    (* ordinary differential equations    *)
      Module[ {clist={},handle,w,i, memo,
	       cleari,clearin, wars={} },
	       
              cleari[fun_[int_]]:=int;
	      cleari[other_]:=other;
	      clearin[L_List]:=Map[cleari,L];
	      Off[Union::heads];
	      handle[Union[rules__,{}]]:=handle[{rules}];
	      handle[li_] :=
			    (* The easier grain                   *)
                Module[{k=1,proceed=True,elist,ww,hrule},
		       hrule[t_[zz]->s_]:=((s-t)==0);
		       Off[Solve::eqf];
		       hrule[Solve[eq_,wrt_]]:=(eq/.(t_[zz]/;
						     MemberQ[{vars},t]):>t);
		       On[Solve::eqf];
		       hrule[Equal[one_,two_]]:=Equal[one,two]/.(t_[zz]/;
						     MemberQ[{vars},t[zz]]):>t;
		       hrule[{homealone_}]:=hrule[homealone];				     
		       While[ (k<=Length[li])&&proceed,
			      elist=Map[hrule,li[[k]]];
			      ww=Solve[elist,clist];
                            (* clist must be externally adjusted  *)
			      If [ (ww=!={}) && FreeQ[ww,Equal],
			           ww=clist/.ww ;
                                   proceed=False,
				   (*else*)
				   k++,
                   Null          ] (*fi*)
                            ] (*elihw*)
			    ;
                       If [ proceed,
			    Infinity,
			    (*else*)
                            ww/.(t_[zz]/;MemberQ[{vars},t[zz]]):>t
                          ] (*fi*)
                            (* So in terms of output we get here *)
			    (* either Infinity or  {<integrals>} *)
                       ] (*eludom*)
		       ;
		       
		       
                 For [ i=1 , i<=Length[{vars}], i++,
		       clist=Append[clist,C[i]];
		       wars=
		       Append[ wars,
		              If[ FreeQ[{vars}[[i]],zz],
			          ({vars}[[i]])[zz],
		                  ({vars}[[i]])
		                ] (*fi*)
			     ]
		       ] (*rof*)
		       ;
		Unprotect[System`$DSolveIntegrals]; (* just in case it got protected *)
		If[ Length[{vars}]<3, (*temporary limitation*)
		    System`$DSolveIntegrals={},
		    Null               (*else nothing as yet*)
		  ];
                w=DSolve[dsystem,wars,zz];
		If[System`$DSolveIntegrals=!={},
		   memo={System`$DSolveIntegrals},
		   (*else*)  
		    If[ FreeQ[w,DSolve] 
		       && FreeQ[w,Infinity] (*tmp*)
		       ,
		       memo=handle[w],
		       (*else*)
		       memo=Infinity
                     ] (*fi*)
		  ] (*fi*) 
               ;
	      If[ memo=!=Infinity,
	          If[ FreeQ[{vars},zz],
		      Null (*nothing*),
		      memo=memo/.(Map[If[FreeQ[#,zz],(#->#[zz]),
		                         (Head[#]->#)]&,{vars}])
		    ] (*fi*)  
	          ,
		  Null (*else nothing*)
		];
		(*fi*)   
	      If[memo===Infinity, memo, Map[clearin,memo]]

	    ] (*eludom*)

pCompleteIntegral[pde_,z_[ivars__],{ivars__},ICr_] :=
   Module[ { ReplaceFormal,U,weq,FreeOfThemAllQ, Formalize, FreeAllIndepQ,
             Separated,SeparateSpecial, KillVariable, EffectSeparation,
	     EasySeparate, EasySeparableQ, PDS, PDSy , OnlyOneVar, Bit,
	     HardHardQ, ClairautQ, ans, count, fold={}, browse, bB},

           ReplaceFormal[formeq_,expr_,{},nN_] := formeq/.(U[]):>expr;
           ReplaceFormal[formeq_,expr_,{vfirst_,vrest___},i_] :=
		  (* Substitute the `expr' instead of the dependent *)
		  (* variable into the  formal equation `formeq'    *)
           ReplaceFormal[formeq/.(U[i]):>(D[expr,vfirst]),expr,{vrest},i+1];
	   
           FreeOfThemAllQ[expr_,{}] := True;
           FreeOfThemAllQ[expr_,{first_,rest___}] := FreeQ[expr,first] &&
					     FreeOfThemAllQ[expr,{rest}];
					     
           Formalize[eq_,u_,{vars__},{},nN_]:=eq/.{u[vars]:>U[]}; 
           Formalize[eq_,u_,{vars__},{vfirst_,vrest___},i_]:=
           (* eq/.Thread[Map[D[u[vars],#]&,{vars}]->Table[U[i],{i,1,Length[{vars}]}]]/.
           u[vars]->U[];*)
           Formalize[(eq/.(D[u[vars],vfirst]:>U[i])),u,{vars},{vrest},i+1];
                  (* U is a global symbol to be further localized *)
		  
           FreeAllIndepQ[equation_,u_,{vars__}]:=
             FreeOfThemAllQ[Formalize[equation,u,{vars},{vars},1],{vars}];

           Separated[sum_,{},nN_] := sum;
           Separated[sum_,{vfirst_,vrest___},i_]:=
             Separated[sum+V[i][vfirst],{vrest},i+1];
                (* Here V is a global symbol to be further localized *)

           EasySeparableQ[expr_,{},{vars__},nN_] := True;
           EasySeparableQ[expr_,{vfirst_,vrest___},{vars__},i_] :=
			    (* Decide if variables are separated in     *)
      			    (* the `expr'                               *)
           FreeOfThemAllQ[ D[expr,vfirst] , Drop[{vars},{i}] ] &&
	        EasySeparableQ[expr,{vrest},{vars},i+1];

           EasySeparate[0,{},stack_] := stack;
           EasySeparate[other_,{},stack_] :=Infinity; 
           EasySeparate[formeq_,{vfirst_,vrest___},stack_] :=
			    (* Try to separate separable equations      *
			     * by mostly syntax analysis                *)
           Module[{split,w},
	     split[a_+b_,var_,flop_]/;FreeQ[a,var] := 
		split[b,var,flop+a];
             split[a_+b_,var_,flop_]/;FreeQ[b,var] :=
		split[a,var,flop+b];
             split[other_,var_,flop_] :=
	     {other,flop};
			    (* this splits out the largest additive part*)
			    (* not containing `var'                     *)
             w = split[Expand[formeq],vfirst,0];
	     If[ FreeOfThemAllQ[w[[1]],{vrest}],
		       EasySeparate[Expand[formeq-w[[1]]],{vrest},
				    Append[stack,w[[1]]] ] ,
                 (*else*)
		 Infinity
               ] (* fi *)
            ] (*eludom*)
            ;

          EffectSeparation[lhslist_,varlist_] :=
          Module[{memo,sumB,flop,i,l},
	     l=Length[varlist];
	     sumB=0; flop=0;
	     For[i=1,i<(l+1),i++,
		 flop=flop+V[i][varlist[[i]]]
		 ] (*rof*)
		 ;
             i=1;
	     While[(i<l) && (flop=!=Infinity),
		   memo = DSolve[lhslist[[i]]==C[l-i+1],V[i][varlist[[i]]],
				 varlist[[i]], GeneratedParameters -> C];
                   memo = memo/.C[1]:>bB[i]; 		
                   If[ Not[FreeQ[memo,Equal]],
		       flop=Infinity ,
		       (*else*)
		       flop= Simplify[Union[Flatten[
				flop/.memo
				]]]
                     ] (*fi*)
		     ;
		     sumB=sumB+C[l-i+1];
		     i++
                  ] (*elihw*)
		  ;
             If[ flop =!= Infinity,
		 If[ FreeQ[lhslist[[l]],Derivative[1][V[l]]],
		     Return[flop/.C->bB/.V[l][varlist[[l]]]->0/.bB[l]->0]
		   ]; (*this really happens: 7/25/94 *)
		 memo = DSolve[lhslist[[l]]+sumB==0,V[l][varlist[[l]]],
			       varlist[[l]], GeneratedParameters -> C];
                 If[ Not[FreeQ[memo,Equal]],
		     flop=Infinity,
		     (*else*)
		     flop= Simplify[Union[Flatten[
			      flop/.memo
			      ]]]
                   ] (*fi*)
		   ,
           Null      (*else nothing*)
               ] (*fi*)
	       ;
	       flop/.C[1]:>bB[l] 			
            ] (*eludom*)
            ;
	    
        KillVariable[formeq_,{vicvar_,vicno_},{vars__}] :=
          Module[{f,wfeq,wfeqx,wlist,wd,i,n,TryKilled,compat},
	     TryKilled[{}] := Infinity;
	     TryKilled[{f1_,fr___}] := 
		Module[{w,c,wl,wi},
                       If[Not[FreeQ[f1,U]],Return[TryKilled[{fr}]]];
		       wi = Integrate[f1,vicvar];
		       w=wfeq/.{f[vicvar]:>
			 (wi+c),f'[vicvar]:>f1};
                       wl = Solve[ Map[(D[w,#]&),{vars}],{c}];
		       If[ (wl === {})||Not[FreeQ[wl,U]],
			   TryKilled[{fr}],
			   (*else*)
			   If[ wl==={{}},
			       wi,
			       (*else*)
			       wi + (c/.wl)[[1]]
                             ] (*fi*)
                         ] (*fi*)
                     ] (*eludom*)
		     ;

	     wfeq=formeq/.
		{U[]:>(U[]+f[vicvar]),U[vicno]:>(U[vicno]+f'[vicvar])};
	     wfeqx = D[wfeq,vicvar];
	     wd= D[wfeqx,U[]];
	     compat = Not[ (wd=!=True) && (FreeQ[wd,f]) ];
	     If [ compat,
                  wlist= { wfeqx, D[wfeqx,U[]] } ;
		  i=1;
		  While[ (i < (Length[{vars}]+1) ) && compat ,
			 wd = D[wfeqx, U[i]];
			 If [ wd === True,
			      Null (*then nothing*)
			      ,
			      (*else*)
			      compat=Not[FreeQ[wd,f]];
			      If [ compat,
				   wlist=Append[wlist,wd],
				   Null (*else nothing*)
                                 ] (*fi*)
                            ] (*fi*)
			    ;
			    i++
                       ] (*elihw*)
		  ,
		  Null (*else nothing*)
                ] (*fi*)
		;
             If [ compat,
                  wlist=Solve[wlist,{f'[vicvar],f''[vicvar]}];
		    (* pending problem of nonlinearity in f' *)
                    (* LXA: 5/12/95: Remember what Chekhov   *)
                    (* [*not* the one from StarTrek] said ?  *)
                    (* "If a rifle is pending on the wall    *)
                    (*  in the beginning of a play, it must  *)
                    (*  eventually shoot" see also bug 13124 *)
	          If[ (!FreeQ[wlist,Solve])||(wlist==={}),
		      Infinity,
		      (*else*)
		      wlist=f'[vicvar]/.wlist;
                      TryKilled[wlist]
                    ] (*fi*)
		    ,
		    (*else*)
		    Infinity
                ] (*fi*)
           ] (*eludom*)
           ;
   
         SeparateSpecial[eq_,{x_,y_}] :=
				 (* Special separation is now available *)
				 (* only for 2 independent variables    *)
           Module[{mem,SepFinish},
	     SepFinish[{}]:={};
	     SepFinish[{{V[1][x]->first_},rest___}]:=
		Module[{me,SF,wme},
			  SF[{},lst_]:={};
			  SF[{{V[2][y]->f_},r___},lst_]:=
			    Union[ Prepend[ SF[{r},lst],
				   {z[ivars] ->
				    (f + (first/.{V[2][y]:>f,      
				     V[2]'[y]:>D[f,y]})) }
				     ] (*dneperp*),
                                           SepFinish[lst]
                                        ] (*noinu*)
					;
                          wme=D[(first/.C[1]:>bB[1]),y]; 
			  If[ Together[wme]===0,
			      Return[
			         Prepend[SepFinish[{rest}],
				    {z[ivars]->(first/.Rule[C[1],bB[1]])}
				         ]],
           		      me=DSolve[wme==0,V[2][y],y, GeneratedParameters -> C]
			    ];
			  me=(me/.C[1]:>bB[2])/.C[2]:>C[1]; 	
			  If[ Not[FreeQ[me,Equal]],
			      Infinity,
			      (*else*)
			      SF[me,{rest}]
                            ] (*fi*)
                        ] (*eludom*)
				 ;
             mem=DSolve[eq, V[1][x] , x, GeneratedParameters -> C];
	     If[ Not[FreeQ[mem,Equal]],
	         Infinity,
	         (*else*)
	         mem=SepFinish[mem];
		 If[ FreeQ[mem,bB[1]], 				
		     Simplify[mem/.C[1]:>bB[1]], 	
		     Simplify[mem] 
                   ] (*fi*)
               ] (*fi*)
             ] (*eludom*)
             ;
		  
   PDS[equation_,u_,{vars__}]/; FreeAllIndepQ[equation,u,{vars}] :=
      Module[ {W,zz,memo,Cit,res},
			    (*`res' is to keep the result*)
	      Cit[sum_,{},nN_]:=sum; 
	      Cit[sum_,{vfirst_,vrest___},i_] :=
		 Cit[sum+bB[i] vfirst,{vrest},i+1]; 

              HardSeparableQ[formeq_,{vv__}] :=
			      (* Special separation is now available *)
			      (* only for 2 independent variables    *)
                 If[Length[{vv}]=!=2,
		    False,
		    (*else*)
		    Simplify[
		       D[formeq,U[1],U[2]] -
		       D[ D[formeq,U[1]] D[formeq,U[2]]/ D[formeq,U[]], U[] ]
			    ] === 0
                   ](*fi*)
		   ;
	      If[ FreeQ[weq,U[]],
		  (*then: the simplest case encountered *)
		  W=Solve[(weq/.{U:>bB}),bB[1]]; 	
		  memo=Cit[0,{vars},1]/.W;
		  res=Map[({z[ivars]->(#+bB[1])}&),memo] 
                  ,
		  (*else*)
		  memo = ReplaceFormal[weq, Separated[0,{vars},1],
				       {vars},1];
                  W=Infinity;
                  If[ EasySeparableQ[memo,{vars},{vars},1],
		      (*then*)
	              W=EasySeparate[memo[[1]],{vars},{}];
                      If[ W=!=Infinity,
			  (*then*)
			  W=EffectSeparation[W,{vars}];
			  If[ W=!=Infinity,
			      Null (*nothing*),
			      (*else*)
			      res=Infinity
                            ] (*fi*)
			    ,
                          (*else*)
			  Null (*nothing*)
                        ],
		      (*else*)
		      W=Infinity
                     ] (*fi*)
		     ;
                  If[ W===Infinity,
		      (*then*)
		      If[ HardSeparableQ[weq[[1]],{vars}] ,
			  (*then*)
			  W=SeparateSpecial[memo,{vars}]
			  ,
			  Null (*else nothing*)
                        ] (*fi*)
			,
            Null (*else nothing*)
                    ]
		    ;
                  If[ W===Infinity,
		      (*then*)
		      W=.;
	              memo = weq/.{U[i_]:>(bB[i] W'[zz]),U[]:>W[zz]};
                      memo = DSolve[memo,W[zz],zz, GeneratedParameters -> C];
	              If[
			  Not[FreeQ[memo,DSolve]],
		          res=PDSFail,
		          (*else*)
		          res=(memo/.{W[zz]:>z[ivars]})/.
		          zz:>Cit[0,{vars},1]
                        ] (*fi*)
			,
			(*else*)
                      If[ Head[W]===List, 
		          res=W,
		          res=Map[{z[ivars]->#}&,W] 
			]
                    ] (*fi*)
                ] (*fi*)
		;
		If[ res === Infinity,
		    PDSFail,
                    If[FreeQ[res,Rule],res=Map[{z[ivars]->#}&,res]];
		    res   (*This is what the module should return *)
                  ] (*fi*)
            ] (*eludom*)
            ;
(*OnlyOneVar is too good to stay inside the Hamilton-Jacobi case  *)

   OnlyOneVar[formeq_,{},nN_] := Infinity ;  
   OnlyOneVar[formeq_,{vfirst_,vrest___},i_] :=
   If[ FreeQ[formeq,vfirst],
       (*then*)
       OnlyOneVar[formeq,{vrest},i+1],
       (*else*)
       If[ FreeOfThemAllQ[formeq,{vrest}],
	   {vfirst,i},
	   Infinity
         ] (*fi*)
     ] (*fi*)
     ;

   Bit[sum_,varlist_]:=
      Module[{wsum,i},
	     wsum=sum;
             For[i=1,i<(Length[varlist]+1),i++,
		 wsum=wsum+bB[i] varlist[[i]] 			
		 ] (*rof*)
		 ;
             Expand[wsum]  
            ] (*eludom*)
            ;
	    
   PDS[ equation_,u_,{vars__}] /; FreeQ[equation,u[vars]] :=
			  (* The case of a Hamilton-Jacobi equation *)
   Module[{w,v,memo,res,HJSeparableQ,HJSeparate  },

          HJSeparableQ[lhs_,vv_] :=
			  (* Special separation is now available *)
			  (* only for 2 independent variables    *)
              Module[ { Fx , Fy , Fp , Fq },
                      If[Length[vv]=!=2,
		         False,
		         (*else*)
			 Fx = D[lhs,vv[[1]]];
			 Fy = D[lhs,vv[[2]]];
			 Fp = D[lhs,U[1]];
			 Fq = D[lhs,U[2]];
			 If[Simplify[
			    Fy Fp D[Fq,vv[[1]]] - D[Fy,vv[[1]]] Fp Fq + 
			    Fx D[Fy,U[1]] Fq  - Fx Fy D[Fq,U[1]]
				    ] === 0 ,
                            (*then*)
			    Simplify[
			    Fx D[Fp,vv[[2]]] Fq - D[Fx,vv[[2]]] Fp Fq +
			    D[Fx,U[2]] Fy Fp - Fx Fy D[Fp,U[2]]
				    ] === 0 ,
                            (*else*)
			    False
                           ] (*fi*)
                       ] (*fi*)
                   ] (*eludom*)
		   ;
          HJSeparate[eq_,{x_,y_}] :=
	     Module[{mem,HJSepFinish},
		    HJSepFinish[{}]:={};
		    HJSepFinish[{{V[1][x]->first_},rest___}] :=
		       Module[{me,HJSF},
	                      HJSF[{},lst_]:={};
			      HJSF[{{V[2][y]->f_},r___},lst_]:=
				Union[ Prepend[ HJSF[{r},lst],
				       {z[ivars] ->
				       (f + (first/.{V[2][y]:>f,
					     V[2]'[y]:>D[f,y]})) }
					     ],
					     HJSepFinish[{rest}]
                                     ] ;
		
			      me=D[(first/.C[1]:>bB[1]),y]; 	
			      me=DSolve[me==0,V[2][y],y, GeneratedParameters -> C];
			      me=(me/.C[i_Integer]:>bB[i]); 	
			      If[ FreeQ[me,Equal],
				  HJSF[me,{rest}],
				  (*else*)
				  Infinity
                                ] (*fi*)
                             ] (*eludom*)
			     ;

		    mem=DSolve[eq,V[1][x],x,GeneratedParameters -> C];
		    If[ FreeQ[mem,Equal],
			(*then*)
			mem=HJSepFinish[mem];
			If[mem==={},
			   Infinity,
			   mem]
			   ,
			(*else*)
			Infinity
                      ] (*fi*)
                   ] (*eludom*)
		   ;

          w=OnlyOneVar[weq,{vars},1];
	  If[ w =!= Infinity,
	      (*then*)
              memo=(weq/.U[w[[2]]]:>v'[w[[1]]])/.U[j_]:>bB[j] ; 
	      memo=DSolve[memo,v[w[[1]]],w[[1]],GeneratedParameters -> C];
	      If[ FreeQ[memo,Equal],
		  (*then*)
		  memo=v[w[[1]]]/.memo; (* !!! *)
		  res=(Map[({z[ivars]->Bit[#,{vars}]}&),memo]/.bB[w[[2]]]->0/.
		      {C[1]:>bB[w[[2]]]}),
		  (*else*)
		  w=Infinity
                ] (*fi*)
		,
              (*else *)
	      memo = ReplaceFormal[weq, Separated[0,{vars},1],
						  {vars},1];
              If[ EasySeparableQ[memo,{vars},{vars},1],
		  (*then*)
		  w=EasySeparate[memo[[1]],{vars},{}];
		  If[ w=!=Infinity,
		      w=EffectSeparation[w,{vars}];
		      If[ w =!= Infinity,
			  res=Map[({z[ivars]->#})&,w]
			  ,
			  Null (*else nothing*)
                        ] (*fi*)
			,
            Null          (*else*)
                    ] (*fi*)
		    ,
                  (*else*)
		  w=Infinity (*temporary logic*)
                ] (*fi*)
            ] (*fi*)
	    ;
          If[ w===Infinity,
	      (*then*)
	      If[ HJSeparableQ[weq[[1]],{vars}],
		  (*then*)
                  w = HJSeparate[memo,{vars}];
		  If[ w=!=Infinity,
		      (*then*)
		      res = w,
              Null (* else nothing *) ] (*fi*)
                  ,
		  Null (*else Pending*)
                ]
		,
        Null     ] (*fi*)
	     ;

          If[ w === Infinity,
	      PDSFail,
	      (*else*)
              res
            ] (*fi*)
         ] (*eludom*)
         ;
	    
       HardHardQ[F_,u_,{x_,y_}]:=
       Module[{p,q,wF,Fx,Fy,Fu,Fp,Fq},
	  wF=(F/.{D[u[x,y],x]:>p,D[u[x,y],y]:>q})/.u[x,y]:>u;
	  Fx=D[wF,x];
	  Fy=D[wF,y];
	  Fu=D[wF,u];
	  Fp=D[wF,p];
	  Fq=D[wF,q];
	  Simplify[Expand[
	     -(Fp Fq D[Fx,y]) - D[Fp,q] Fx Fy + Fp D[Fx,q]  Fy + Fq Fx D[Fy,p] -
	     D[Fp,q]  Fu Fy p +Fp D[Fu,q] Fy p + Fq Fu D[Fy,p] p - Fp Fq*
	     D[Fy,u] p - D[Fp,q] Fu Fx q + Fq D[Fu,p] Fx q +
	     Fp Fu D[Fx,q] q - Fp Fq D[Fx,u]  q - D[Fp,q] Fu^2  p q + Fq Fu *
	     D[Fu,p] p q + Fp Fu D[Fu,q] p q - Fp Fq D[Fu,u] p q
	     ]] === 0
         ] (*eludom*)
         ;
       HardHardQ[other__]:=False;

       ClairautQ[varlist_]:=
          Module[{i},
	     i=1;
	     While[ (i<=Length[varlist]) &&
	        (Simplify[Expand[D[weq[[1]],varlist[[i]]]+
		U[i] D[weq[[1]],U[]]]]===0) ,
		i++
		  ] (*elihw*)
		  ;
		  i===l
            ](*eludom*)
            ;

   PDS[equation_,u_,{vars__}]/;ClairautQ[{vars}] :=
      Module[{w,A,wu,i},
	     For[i=1,i<(Length[{vars}]+1),i++,
		 weq=weq/.{U[i]:>bB[i]}			
		 ] (*rof*)
		 ;
	     wu=Bit[A,{vars}];
	     w=Solve[Simplify[Expand[(weq/.U[]:>wu)]],A];
             If[ FreeQ[w,Equal] &&
		 FreeOfThemAllQ[w,{vars}],
		 Map[({z[ivars]->#})&,wu/.w],
		 (*else*)
		 PDSFail
                 (*temporary logic*)
               ]
            ] (*eludom*)

            ;
   PDS[equation_,u_,{vars__}]:=  
   Module[ {w,key,fin1,memo,Den,v,i,Num,Scaleit,ScaleCoef},
	   fin1[{z[ivars]->term_}]:={z[ivars]->term+key};
	   fin1[other_]:={u[ivars]->other+key};
	   Scaleit[lhs_,var_] :=
	      Module[{wlhs,mem,g},
		     wlhs=lhs/.var[]:>v;
		     For[i=1,i<(Length[{vars}]+1),i++,
			 If[i===w[[2]],
			    wlhs=wlhs/.var[i]:>(g/(w[[1]])),
			    wlhs=wlhs/.var[i]:>(bB[i] g)	
                           ] (*fi*)
			   ] (*rof*)
			   ;
                     mem=Solve[Simplify[Expand[wlhs]]==0,g];
		     If[ Not[mem==={}] && FreeQ[mem,Equal],
			 mem=g/.mem;
			 mem=Map[(Integrate[1/#,{v,1,z[ivars]}]==
				 Log[bB[w[[2]]] w[[1]] ] + 
				 Bit[-bB[w[[2]]] w[[1]] , {vars}])&, mem]; 
                         If[ FreeQ[mem,Integrate],
			     Map[({#})&,
			     Flatten[Map[(Solve[#,z[ivars]])&,mem]]], 
			     (*else*)
			     mem
                           ] (*mem*)
			   ,
                         (*else*)
			 Infinity
                       ] (*fi*)

		    ] (*eludom*)
		    ;
           
	   ScaleCoef:= 
	      Module[ {a,b,i,wlst,we,wwe,compt},
				    (*Num,Den assumed to be global here*)
                      we= Num + a Den == b weq[[1]] ;
		      wwe = D[we,U[]];
		      compt = Not[(wwe=!=True) && FreeQ[wwe,a] && FreeQ[wwe,b]];
		      If [ compt,
		           If [ wwe === True,
			        wlst={we},
			        wlst={we,wwe}
                             ] (*fi*)
			   ;
                           While[(i< (Length[{vars}]+1))&&compt,
			         wwe=D[we,U[i]];
			         If[ wwe===True,
				     Null (*then nothing*),
				     (*else*)
				     compt=Not[FreeQ[wwe,a] && FreeQ[wwe,b]];
				     If [ compt,
			                  wlst=Append[wlst,wwe],
			                  Null (*else nothing*)
                                        ] (*fi*)
                                    ] (*fi*)
				    ;
				    i++
                                 ] (*elihw*)
				 ;
                           If [ compt,
				wlst=Solve[wlst,{a,b}];
				wlst=a/.wlst;
				If [ wlst === {},
				     Infinity,
				     a=Simplify[Expand[wlst[[1]]]];
				     If[ FreeQ[a,U] && FreeQ[a,w[[1]]],
				         a,
				         Infinity
					 ] (*fi*)
                                  ] (*fi*)
				  ,
                                Infinity
                              ] (*fi*)
			      ,
                           Infinity
                        ] (*fi*)
                 ] (*eludom*)
		 ;

	   w=OnlyOneVar[weq,{vars},1];
	   If[ w=!=Infinity,
	       (*then*)
	       key=KillVariable[weq,w,{vars}];
	       If[ key=!=Infinity,
		   (*then*)
		   weq=Simplify[Expand[
		       equation/.{u[vars]:>(u[vars]+key),
			  D[u[vars],w[[1]]]:>D[(u[vars]+key),w[[1]]]}
                       ]]
		       ;
		       (*correctness check Pending *)
		   weq = Formalize[(Simplify[weq[[1]]-weq[[2]]]==0),u,{vars},{vars},1];
		   weq=PDS[ weq,u,{vars}]
		   ;
		   If[ FreeQ[weq,Equal] && FreeQ[weq,PDSFail],
		       Map[fin1,weq],
		       (*else*)
		       PDSFail
                     ]
                     ,
		   (*else*)
		   Den=Simplify[Expand[
		      w[[1]]D[weq[[1]],w[[1]]]-U[w[[2]]] D[weq[[1]],U[w[[2]]]]
			       ]];
                   If[ Den===0,
		       (*then*)
				(* It is sufficient for the equation to have *)
				(* the y/dy symmetry. It is not the general  *)
				(* case of such symmetry however unless the  *)
				(* equation is not reasonably *factored*     *)
		       memo=Scaleit[weq[[1]],U];
		       If[ memo =!= Infinity,
                           memo 
			   ,
			   (*else*)
			   PDSFail (*temporary logic*)
                         ] (*fi*)
                       ,
		       (*else, Den =!= 0 *)
                       Num=U[] D[weq[[1]],U[]];
		       For[i=1,i<(Length[{vars}]+1),i++,
			   Num=Num+U[i] D[weq[[1]],U[i]]
                          ] (*rof*)
			  ;
                       Num=Simplify[Expand[Num]];
		       If[ Num === 0,
			   (*then, u d/du is a symmetry*) 
			   key=weq;
			   For[i=1,i<(Length[{vars}]+1),i++,
			   If[i===w[[2]],
			      key=key/.U[i]:>(v U[]),
			      key=key/.U[i]:>(bB[i] U[])	
                             ] (*fi*)
			     ] (*rof*)
			     ;
			     memo=
                           Solve[Simplify[Expand[key]],v];
			   If [ FreeQ[memo,Equal],
				memo=v/.memo;
				memo=
                                Map[
				   ( Exp[Bit[Integrate[#,w[[1]]] + bB[w[[2]]]-
					      bB[w[[2]]] w[[1]], {vars} ] ] )&,
                                    memo ];			
                                Map[ ({z[ivars]->#})&,memo]
				,
				(*else*)
				PDSFail (*temporary logic*)
                              ] (*fi*)
                           ,
			   (*else, Num=!=0 *)
			   key=ScaleCoef;
			   If[ key =!= Infinity,
			       (*then*)
			       key=1/key;
			       memo=weq[[1]]/.U[]:>(V[] w[[1]]^key);
			       For[i=1,i<(Length[{vars}]+1),i++,
				   If[ i===w[[2]],
				       memo=memo/.U[i]:>
					  (w[[1]]^key V[i] +
					   key w[[1]]^(key-1) V[]),
                                       memo=memo/.U[i]:>(w[[1]]^key V[i])
                                     ] (*fi*)
				     ] (*rof*)
				     ;

			       memo= Scaleit[Simplify[Expand[memo]],V];
                               If[ memo=!=Infinity,
				   (*then*)
                                   Map[ ({z[ivars]->
					  Simplify[#[[1]][[2]] w[[1]]^key]})&,
					  memo] ,
                                   (*else*)
				   PDSFail (*temporary logic*)
                                 ] (*fi*)
				 ,
				 (*else*)
                               PDSFail     (*temporary logic*)
                             ] (*fi*)
                          ] (*fi, Num=!=0 clause      *)
                       ] (*fi, Den=!=0 clause         *)
                    ] (*fi, variable not killed clause*)  
		 ,
		 (*else: not only one variable*)
               memo = ReplaceFormal[weq, Separated[0,{vars},1],
				    {vars},1];
               If[ EasySeparableQ[memo,{vars},{vars},1],
		   (*then*)
                   w=EasySeparate[memo[[1]],{vars},{}];
		   If[ w=!=Infinity,
		       w=EffectSeparation[w,{vars}];
                       If[ w =!= Infinity,
			   Map[({z[ivars]->#})&,w],
			   (*else*)
			   PDSFail (*temporary logic*)
                         ] (*fi*)
			 ,
                       (*else*)
		       PDSFail (*temporary logic*)
                     ] (*fi*)
		     ,
                   (*else*)

		   If[ HardHardQ[equation[[1]],u,{vars}],
		       w=SeparateSpecial[memo,{vars}];
		       If[ w === Infinity,
		           PDSFail,
			   (*else*)
			   w
			 ]  
		       ,
		       (*else*)
		       PDSFail
                     ] (*fi*)
                 ] (*fi*)
	      ] (*fi*)
          ] (*eludom*)
          ;
	  
   PDSy[D[u_[x_,y_],y_]==F_,u_,{x_,y_}] := 
   Module[{p,q,f, fp, fx, Den1, Num1, v,
	   key1 , key2 , keyaux, ScaleChar1, ScaleChar2, res, ans },
			   (* Our aim is to find 2 commuting symmetries *)
          ScaleChar1 := 
	     Module[ {a},
		     a=Together[- Num1/Den1]; 
                     If[ FreeQ[a,x] && FreeQ[a,y] &&
			 FreeQ[a,u] && FreeQ[a,p] ,
			 (* then *)
			 y q + a x p ,
			 (*else*)
			 Infinity
                       ] (*fi*)
                   ] (*eludom*)
		   ;

          ScaleChar2 :=
	     Module [ {a,b,R,Den2,Effect},
		           (* we assume here that y d/dy is not admitted *)
			   (* and therefore  (f+y fy)  =!= 0             *)
                      Effect[v_] :=
			 Module[{},
				a= Simplify[Expand[
				      (R D[Num1,v] - D[R,v] Num1)/Den2
				      ]];
                                If[ FreeQ[a,u] && FreeQ[a,p] &&
				    FreeQ[a,x] && FreeQ[a,y],
				    b= Simplify[Expand[
					  (Den1 D[R,v] - D[Den1,v] R)/Den2
					  ]];
                                    If[ FreeQ[b,u] && FreeQ[b,p] &&
					FreeQ[b,x] && FreeQ[b,y],
					u - a p - b q,
					(*else*)
					Infinity
                                      ] (*fi*)
				      ,
				      (*else*)
                                    Infinity
                                  ] (*fi*)
                               ] (*eludom*)
			       ;

                      R= Together[ f - u D[f,u] - p fp ] ;
                      If[ R === 0,
			  u, 
			   (* OK  u d/du is symmetry: no further search  *)
			   (* needed                                     *)
                           (*else*)
                          Den2 = Simplify[Expand[
				    Den1 D[Num1,u] - Num1 D[Den1,u]
				    ]];
                          If[ Den2 === 0,
			      Den2 = Simplify[Expand[
				 Den1 D[Num1,p] - Num1 D[Den1,p]
				 ]];
                              If[ Den2 === 0,
				  Den2 = Simplify[Expand[
				     Den1 D[Num1,x] - Num1 D[Den1,x]
				     ]];
                                  If[ Den2 === 0,
				      Den2 = Simplify[Expand[
					 Den1 D[Num1,y] Num1 D[Den1,y]
					 ]];
                                      If[ Den2 === 0,
					  Infinity,
					  Effect[y]
                                        ] (*fi*)
					,
					(*else*)
                                      Effect[x]
                                    ] (*fi*)
				    ,
				    (*else*)
                                  Effect[p]
                                ] (*fi*)
				,
                              Effect[u]
                            ] (*fi*)
                         ] (*fi*)
                    ] (*eludom*)
		    ;

	  UseSpecial[xx_,yy_,ff_,uu_,pp_] :=
			     (* Assume that xx d/dxx and yy d/dyy are *)
			     (* symmetries                            *)
             Module[{g,w,Completeit} ,
		    Completeit[{}]:={};
		    Completeit[{first_,rest___}] :=
		       Module[{me},
			      me=Solve[
				    Integrate[1/first,{uu,1,z[ivars]}]
				    == (Log[bB[1] xx] + bB[2] Log[yy] ), 
				    z[ivars]
				      ];
				  Simplify[Union[ me , Completeit[{rest}]]]
                             ] (*eludom*)
			     ;

		    w= Solve[Simplify[Expand[(ff yy)/.pp:>(g/xx)]]
			     == (bB[2] g), g ];			
                    If[ FreeQ[w,Equal],
			w = g/.w;
			w=Completeit[w];
			If[ w==={},
			    Infinity,
			    w
			  ] (*fi*)  
			    ,
			(*else*)
			Infinity
                      ]
                   ] (*eludom*)
		   ;

          
          UseSymmetries[sym1_,sym2_]/;FreeQ[{sym1,sym2},Infinity] :=
	     Module[{mem,w, Completeit},
		    Completeit[{}]:={};
		    Completeit[{first_,rest___}]:=     
		       Module[{me},
			      me=DSolve[ Simplify[
				    D[first,y]==((f/.p:>D[first,x])/.u:>first)
				    ] ,
					w[y],y, GeneratedParameters -> C];

                              If[ FreeQ[me,Equal],
				  me=first/.me;
				  me=me/.C[1]:>bB[2];		
				  Union[
				     Map[({z[ivars]->#})&,me],
				     Completeit[{rest}]
				       ],
                                  (*else*)
				  Completeit[{rest}]
                                ] (*fi*)
                             ] (*eludom*)
			     ;

		    mem = DSolve[ ((sym2== bB[1] sym1)/.q:>f)/.	
				    {u:>w[x],p:>w'[x]},w[x],x,GeneratedParameters -> C];
                    If[ FreeQ[mem,Equal],
			mem=(w[x]/.mem)/.C[1]:>w[y] ;
			mem=Completeit[mem];
			If[ mem==={},
			    Infinity,
			    mem
                          ] (*fi*)
			  ,
			  (*else*)
                        Infinity
                      ] (*fi*)
                 ] (*eludom*)
		 ;
          
	  UseSymmetries[otherwise__]/;True:= Infinity
	  ;
	  
	  f=(F/.{D[u[x,y],x]:>p})/.u[x,y]:>u;
	  fp = D[f,p];
	  fx = D[f,x];
	  key1=Infinity;
	  res=Infinity;
	  Den1 = Together[- p fp + x fx];
	  Num1= Together[f+y D[f,y]];
	  If[ Den1 === 0,
	      (*then*)
	      key1= x p;
	      If[ Num1 === 0,
		  (*then*)
		  key2 = y q; (* OK, y/dy is a symmetry *)
		  res = UseSpecial[x,y,f,u,p]
                  ,
		  (*else*)
	           key2 = Simplify[Expand[
			     (f-u D[f,u]-p fp)/Num1
			     ]] ;
                  If[ FreeQ[key2,u] && FreeQ[key2,p] &&
		      FreeQ[key2,x] && FreeQ[key2,y],
		      (* key2 = u - key2 y q;
		      res = UseSymmetries[key1,key2]*)
		      key2 = 1/key2 ; (*alpha*)
		      res=
			 UseSpecial[x,y,
			    Simplify[Expand[((f y^(-key2))/.
			       {u:>v y^key2,p:>p y^key2}) - key2 v / y ]],
				   v, p];
                      If[ FreeQ[res,Equal],
			  res= res/.
			       ({z[ivars]->any_}):>
			       ({z[ivars]->Simplify[y^key2 any]}),
                          (*else*)
			  res=res/.z[ivars]:>(z[ivars]/y^key2)
                        ] (*fi*)
		      ,
		      (*else*)
		      key2 = Infinity;
		      res=Infinity
                    ] (*fi*)
                ] (*fi*)
	      ,  (* the simplest scaling x d/dx is admitted   *)
	      (*else*)
	      key1 = ScaleChar1; 
              If[ Num1 === 0,    (* Implies that key1 === y q *)
		  (*then*)
		  key2 = Simplify[Expand[
			    (f-u D[f,u]-p fp)/Den1
			    ]] ;
                  If[ FreeQ[key2,u] && FreeQ[key2,p] &&
		      FreeQ[key2,x] && FreeQ[key2,y],
		      key2 = u - key2 x p;
		      res = UseSymmetries[key1,key2]
		      ,
		      (*else*)
		      key2 = Infinity;
		      res=Infinity
                    ] (*fi*)
		    ,
		    (*else*)
                  key2 = ScaleChar2
                ] (*fi*)
            ] (*fi*);
	    If[ FreeQ[res,Infinity],
	        res,
	        res=UseSymmetries[key1,key2];
		If[ FreeQ[res,Infinity],
		    res,
		    PDSFail
		  ]  	
              ]

	 ] (*eludom*)
	 ;
	   bB=GeneratedParameters/.ICr;  	
           dbgl=Map[D[z[ivars],#]&,{ivars}];
           dbgr=Table[U[foo],{foo,1,Length[{ivars}]}];
           dbgt=Thread[dbgl->dbgr];
           dbgu=z[ivars];
           weq=(((pde[[1]]-pde[[2]])==0)/.dbgt)/.dbgu->U[];	    
           (* weq = Formalize[(pde[[1]]-pde[[2]])==0,z,{ivars},{ivars},1]; *)
           ans= PDS[pde,z,{ivars}];
           If[ FreeQ[ans,PDSFail],
	       Null (*then nothing*)
	       ,
	       If[ Length[{ivars}]==2,
	           ans=Solve[pde,(D[z[ivars],{ivars}[[2]]])];
	            If[ FreeQ[ans,Solve]&&(ans=!={})&&(ans=!={{}}),
		       ans=(D[z[ivars],{ivars}[[2]]])/.ans;
		       For[ count=1,count<=Length[ans],count++,
		            browse=PDSy[(D[z[ivars],{ivars}[[2]]])==ans[[count]],
			                z,{ivars}];
			    If[ FreeQ[browse,PDSFail],
			        fold=Union[fold,browse],
				Null (*else nothing*)
			      ]	 		
		          ] (*rof*)
			  ;
		       If[ fold==={},
		           ans=PDSFail,
			   (*else*)
			   ans=fold
			 ] (*fi*)  	  
		       ,
		       (*else*)
		       ans=PDSFail
		     ] (*fi*)
		     (*****************************************************)
		     (* Algorithm design needs improvement at this point  *)
		     (*****************************************************)
		       ,
		       Null (*else nothing*)
		     ] (*fi length =!= 2*)    		   
		 ] (*fi*)
             ;
	   ans/;FreeQ[ans,PDSFail]   

         ] /; PDEAppl1[pde,z[ivars],{ivars}](*eludom*)
	 
(*========================================================================*)


 End[ ] (* "Calculus`PDSolve1`Private` *)

 EndPackage[ ] (* "Calculus`PDSolve1`" *)

